
function [xyw_bar,pointset_stats]=set_dunavant_C_barycentric(deg)

%--------------------------------------------------------------------
% CUBATURE RULE ON THE SIMPLEX [0 0; 0 1; 1 0] WITH ADE EQUAL TO deg.
% THE CUBATURE RULE IS DESCRIBED IN BARYCENTRICAL FORM.
%--------------------------------------------------------------------
% INPUT:
% deg: algebraic degree of precision (ADE) of the rule.
%--------------------------------------------------------------------
% OUTPUT:
% xyw_bar: MATRIX OF NODES xyw(:,1:3) IN BARYCENTRICAL COOR. AND WEIGHTS xyw(:,4).
%      REMARK: The sum of the weights makes 1.
%
% pointset_stats:
%    statistics of the pointset stored in a matrix. Each rows contains
%    degree, cardinality of the full pointset, moment error w.r.t.
%    orthn. Dubiner basis, number of points with negative weights,
%    number of points outside the domain.
%--------------------------------------------------------------------
% SOURCE:
%  D.A. Dunavant,
%  "High Degree Efficient Symmetrical Gaussian Quadrature Rules for the
%  Triangle"
%  International Journal for Numerical Methods in Engineering,
%  21(6), June 1985, pp. 1129-1148.
%--------------------------------------------------------------------
% | DEG | CARD | W- | PT.O|   M.E.   |Q.|
%--------------------------------------------------------------------
% |   1 |   1 |   0 |   0 | 1.92e-16 |PI|
% |   2 |   3 |   0 |   0 | 3.68e-16 |PI|
% |   3 |   4 |   1 |   0 | 2.22e-16 |NI|
% |   4 |   6 |   0 |   0 | 2.01e-16 |PI|
% |   5 |   7 |   0 |   0 | 3.15e-16 |PI|
% |   6 |  12 |   0 |   0 | 3.12e-16 |PI|
% |   7 |  13 |   1 |   0 | 3.75e-16 |NI|
% |   8 |  16 |   0 |   0 | 6.11e-16 |PI|
% |   9 |  19 |   0 |   0 | 3.19e-16 |PI|
% |  10 |  25 |   0 |   0 | 5.07e-16 |PI|
% |  11 |  27 |   0 |   3 | 8.88e-16 |P0|
% |  12 |  33 |   0 |   0 | 3.89e-16 |PI|
% |  13 |  37 |   0 |   0 | 4.79e-16 |PI|
% |  14 |  42 |   0 |   0 | 8.47e-16 |PI|
% |  15 |  48 |   0 |   9 | 8.47e-16 |P0|
% |  16 |  52 |   0 |   6 | 9.02e-16 |P0|
% |  17 |  61 |   0 |   0 | 5.64e-16 |PI|
% |  18 |  70 |   3 |   6 | 1.11e-15 |N0|
% |  19 |  73 |   0 |   0 | 8.10e-16 |PI|
% |  20 |  79 |   3 |   9 | 9.91e-16 |N0|
%--------------------------------------------------------------------
% LEGEND:
%--------------------------------------------------------------------
% DEG  : algebraic degree of precision
% CARD : cardinality of the pointset
% W-   : negative weights
% PT.O : points outside the simplex [0 0; 1 0; 0 1].
% M.E. : maximum moment error in inf norm.
% Q.: quality of the rule.
%        PI: pos. weights, points in simplex
%        NI: some neg. weights, points in simplex
%        P0: pos. weights, some points out of the simplex
%        N0: some neg. weights, some points out of the simplex
%--------------------------------------------------------------------



switch deg
    
    case   1
        
        % ALG. DEG.:    1
        % PTS CARD.:    1
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.9e-16
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333425851918718763045e-01 1.00000000000000000000000000000000e+00
            ];
        
    case   2
        
        % ALG. DEG.:    2
        % PTS CARD.:    3
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.7e-16
        
        xyw_bar=[
            1.66666666666666657414808128123695e-01 1.66666666666666657414808128123695e-01 6.66666666666666740681534975010436e-01 3.33333333333333314829616256247391e-01
            1.66666666666666657414808128123695e-01 6.66666666666666740681534975010436e-01 1.66666666666666629659232512494782e-01 3.33333333333333314829616256247391e-01
            6.66666666666666740681534975010436e-01 1.66666666666666657414808128123695e-01 1.66666666666666601903656896865868e-01 3.33333333333333314829616256247391e-01
            ];
        
    case   3
        
        % ALG. DEG.:    3
        % PTS CARD.:    4
        % NEG. W.  :    1
        % OUT PTS. :    0
        % M.E.INF. : 2.2e-16
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333425851918718763045e-01 -5.62500000000000000000000000000000e-01
            2.00000000000000011102230246251565e-01 2.00000000000000011102230246251565e-01 6.00000000000000088817841970012523e-01 5.20833333333333370340767487505218e-01
            2.00000000000000011102230246251565e-01 5.99999999999999977795539507496869e-01 2.00000000000000066613381477509392e-01 5.20833333333333370340767487505218e-01
            5.99999999999999977795539507496869e-01 2.00000000000000011102230246251565e-01 2.00000000000000011102230246251565e-01 5.20833333333333370340767487505218e-01
            ];
        
    case   4
        
        % ALG. DEG.:    4
        % PTS CARD.:    6
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 2.0e-16
        
        xyw_bar=[
            4.45948490915965001235576892213430e-01 4.45948490915964779190971967182122e-01 1.08103018168070219573451140604448e-01 2.23381589678011471811203136894619e-01
            4.45948490915965001235576892213430e-01 1.08103018168070233451238948418904e-01 4.45948490915964779190971967182122e-01 2.23381589678011471811203136894619e-01
            1.08103018168070219573451140604448e-01 4.45948490915965001235576892213430e-01 4.45948490915964779190971967182122e-01 2.23381589678011499566778752523533e-01
            9.15762135097707846709269574603240e-02 9.15762135097707846709269574603240e-02 8.16847572980458513924872931966092e-01 1.09951743655321995674079005311796e-01
            9.15762135097708124265025730892376e-02 8.16847572980458402902570469450438e-01 9.15762135097707430375635340169538e-02 1.09951743655321884651776542796142e-01
            8.16847572980458624947175394481746e-01 9.15762135097706875264123027591268e-02 9.15762135097706875264123027591268e-02 1.09951743655321787507261888094945e-01
            ];
        
    case   5
        
        % ALG. DEG.:    5
        % PTS CARD.:    7
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.2e-16
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333370340767487505218e-01 3.33333333333333370340767487505218e-01 2.25000000000000005551115123125783e-01
            4.70142064105115164984738385101082e-01 4.70142064105115109473587153843255e-01 5.97158717897697255416744610556634e-02 1.32394152788506191953388224646915e-01
            4.70142064105115053962435922585428e-01 5.97158717897698365639769235713175e-02 4.70142064105115109473587153843255e-01 1.32394152788506191953388224646915e-01
            5.97158717897698157472952118496323e-02 4.70142064105115109473587153843255e-01 4.70142064105115109473587153843255e-01 1.32394152788506164197812609018001e-01
            1.01286507323456301254971378966729e-01 1.01286507323456342888334802410100e-01 7.97426985353087314223330395179801e-01 1.25939180544827167285149016606738e-01
            1.01286507323456342888334802410100e-01 7.97426985353087314223330395179801e-01 1.01286507323456342888334802410100e-01 1.25939180544827195040724632235651e-01
            7.97426985353087203201027932664147e-01 1.01286507323456356766122610224556e-01 1.01286507323456440032849457111297e-01 1.25939180544827167285149016606738e-01
            ];
        
    case   6
        
        % ALG. DEG.:    6
        % PTS CARD.:   12
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.1e-16
        
        xyw_bar=[
            2.49286745170910012392440080475353e-01 2.49286745170912538149821102706483e-01 5.01426509658177477213314432447078e-01 1.16786275726379243367070159820287e-01
            2.49286745170910012392440080475353e-01 5.01426509658180030726271070307121e-01 2.49286745170909984636864464846440e-01 1.16786275726378854789011541015498e-01
            5.01426509658180030726271070307121e-01 2.49286745170910012392440080475353e-01 2.49286745170909956881288849217526e-01 1.16786275726380089912126436502149e-01
            6.30890144915022543781191188827506e-02 6.30890144915020878446654251092696e-02 8.73821971016995657777215456007980e-01 5.08449063702067424741493084638932e-02
            6.30890144915021294780288485526398e-02 8.73821971016995768799517918523634e-01 6.30890144915020600890898094803561e-02 5.08449063702066869629980772060662e-02
            8.73821971016995213688005605945364e-01 6.30890144915025180560874673574290e-02 6.30890144915022682559069266972074e-02 5.08449063702071032966323116397689e-02
            5.31450498448169314502642635034135e-02 3.10352451033784004774673803694895e-01 6.36502499121399001325016797636636e-01 8.28510756183735846969184990484791e-02
            5.31450498448167510390227619154757e-02 6.36502499121398890302714335120982e-01 3.10352451033784393352732422499685e-01 8.28510756183734597968282287183683e-02
            3.10352451033784004774673803694895e-01 5.31450498448177224841693089274486e-02 6.36502499121398224168899560027057e-01 8.28510756183745561420650460604520e-02
            3.10352451033783893752371341179241e-01 6.36502499121399001325016797636636e-01 5.31450498448171604337630924419500e-02 8.28510756183735708191306912340224e-02
            6.36502499121398557235806947574019e-01 5.31450498448165498110995486058528e-02 3.10352451033784892953093503820128e-01 8.28510756183731406077086489858630e-02
            6.36502499121397891101992172480095e-01 3.10352451033785337042303353882744e-01 5.31450498448167718557044736371608e-02 8.28510756183732655077989193159738e-02
            ];
        
    case   7
        
        % ALG. DEG.:    7
        % PTS CARD.:   13
        % NEG. W.  :    1
        % OUT PTS. :    0
        % M.E.INF. : 3.7e-16
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333425851918718763045e-01 -1.49570044467681767530464753690467e-01
            2.60345966079039703444664155540522e-01 2.60345966079039925489269080571830e-01 4.79308067841920371066066763887648e-01 1.75615257433207827553189872560324e-01
            2.60345966079039981000420311829657e-01 4.79308067841920260043764301371993e-01 2.60345966079039814466966618056176e-01 1.75615257433207883064341103818151e-01
            4.79308067841920426577217995145475e-01 2.60345966079039647933512924282695e-01 2.60345966079039981000420311829657e-01 1.75615257433207772042038641302497e-01
            6.51301029022158389780727816287254e-02 6.51301029022158112224971659998118e-02 8.69739794195568349799430052371463e-01 5.33472356088385002426122127872077e-02
            6.51301029022158112224971659998118e-02 8.69739794195568349799430052371463e-01 6.51301029022158806114362050720956e-02 5.33472356088384933037183088799793e-02
            8.69739794195568349799430052371463e-01 6.51301029022158389780727816287254e-02 6.51301029022158112224971659998118e-02 5.33472356088384933037183088799793e-02
            4.86903154253163772402501763281180e-02 3.12865496004873788749023333366495e-01 6.38444188569809778499575259047560e-01 7.71137608902571214386512110650074e-02
            4.86903154253164327514014075859450e-02 6.38444188569809667477272796531906e-01 3.12865496004873899771325795882149e-01 7.71137608902572185831658657662047e-02
            3.12865496004873899771325795882149e-01 4.86903154253164396902953114931734e-02 6.38444188569809667477272796531906e-01 7.71137608902572047053780579517479e-02
            3.12865496004873844260174564624322e-01 6.38444188569809778499575259047560e-01 4.86903154253164327514014075859450e-02 7.71137608902571353164390188794641e-02
            6.38444188569809778499575259047560e-01 4.86903154253163703013562724208896e-02 3.12865496004873844260174564624322e-01 7.71137608902570936830755954360939e-02
            6.38444188569809667477272796531906e-01 3.12865496004873899771325795882149e-01 4.86903154253164327514014075859450e-02 7.71137608902572324609536735806614e-02
            ];
        
    case   8
        
        % ALG. DEG.:    8
        % PTS CARD.:   16
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.1e-16
        
        xyw_bar=[
            3.33333333333333370340767487505218e-01 3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 1.44315607677787116625012231452274e-01
            4.59292588292722903098308506741887e-01 4.59292588292723402698669588062330e-01 8.14148234145536942030219051957829e-02 9.50916342672846193195823616406415e-02
            4.59292588292723180654064663031022e-01 8.14148234145535970585072504945856e-02 4.59292588292723180654064663031022e-01 9.50916342672845776862189381972712e-02
            8.14148234145537913475365598969802e-02 4.59292588292723014120610969257541e-01 4.59292588292723236165215894288849e-01 9.50916342672846609529457850840117e-02
            1.70569307751760351754555244951916e-01 1.70569307751760018687647857404954e-01 6.58861384496479574046645666385302e-01 1.03217370534718258512363320278382e-01
            1.70569307751760157465525935549522e-01 6.58861384496479685068948128900956e-01 1.70569307751760157465525935549522e-01 1.03217370534718258512363320278382e-01
            6.58861384496479463024343203869648e-01 1.70569307751760323998979629323003e-01 1.70569307751760212976677166807349e-01 1.03217370534718286267938935907296e-01
            5.05472283170309635846884077636787e-02 5.05472283170309844013701194853638e-02 8.98905543365938086708410992287099e-01 3.24584976231980723571091118628829e-02
            5.05472283170309219513249843203084e-02 8.98905543365938086708410992287099e-01 5.05472283170309566457945038564503e-02 3.24584976231980723571091118628829e-02
            8.98905543365938086708410992287099e-01 5.05472283170309913402640233925922e-02 5.05472283170309219513249843203084e-02 3.24584976231980792960030157701112e-02
            8.39477740995770924103958066098130e-03 2.63112829634638223375020515959477e-01 7.28492392955404022281129527982557e-01 2.72303141744350517272632572485236e-02
            8.39477740995758260622583435406341e-03 7.28492392955404022281129527982557e-01 2.63112829634638445419625440990785e-01 2.72303141744349996855589779443108e-02
            2.63112829634638001330415590928169e-01 8.39477740995748025754075172244484e-03 7.28492392955404466370339378045173e-01 2.72303141744349060104912751967277e-02
            2.63112829634637945819264359670342e-01 7.28492392955404577392641840560827e-01 8.39477740995753229924503102665767e-03 2.72303141744349268271729869184128e-02
            7.28492392955404466370339378045173e-01 8.39477740995771617993348456820968e-03 2.63112829634637834796961897154688e-01 2.72303141744350447883693533412952e-02
            7.28492392955404022281129527982557e-01 2.63112829634638223375020515959477e-01 8.39477740995775434384995605796576e-03 2.72303141744350447883693533412952e-02
            ];
        
    case   9
        
        % ALG. DEG.:    9
        % PTS CARD.:   19
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.2e-16
        
        xyw_bar=[
            3.33333333333332537673499018637813e-01 3.33333333333333925452279800083488e-01 3.33333333333333481363069950020872e-01 9.71357962827986559650739195603819e-02
            4.89682519198738286370087280374719e-01 4.89682519198736898591306498929043e-01 2.06349616025248150386062206962379e-02 3.13347002271392102712432858879765e-02
            4.89682519198738008814331124085584e-01 2.06349616025247942219245089745527e-02 4.89682519198737231658213886476005e-01 3.13347002271391478211981507229211e-02
            2.06349616025248254469470765570804e-02 4.89682519198737342680516348991659e-01 4.89682519198737786769726199054276e-01 3.13347002271390784322591116506374e-02
            4.37089591492935969352373604124296e-01 4.37089591492937024064246998023009e-01 1.25820817014126951072228166594869e-01 7.78275410047742505215140340624203e-02
            4.37089591492936135885827297897777e-01 1.25820817014127089850106244739436e-01 4.37089591492936802019642072991701e-01 7.78275410047744309327555356503581e-02
            1.25820817014126506983018316532252e-01 4.37089591492936968553095766765182e-01 4.37089591492936524463885916702566e-01 7.78275410047740145991213012166554e-02
            1.88203535619032219505797343117592e-01 1.88203535619032996661914580727171e-01 6.23592928761934728321136844897410e-01 7.96477389272102070938430529167817e-02
            1.88203535619032163994646111859765e-01 6.23592928761935172410346694960026e-01 1.88203535619032691350582808809122e-01 7.96477389272101377049040138444980e-02
            6.23592928761933729120414682256524e-01 1.88203535619033246462095121387392e-01 1.88203535619033024417490196356084e-01 7.96477389272103319939333232468925e-02
            4.47295133944527190106299485705676e-02 4.47295133944527051328421407561109e-02 9.10540973211094506467588871601038e-01 2.55776756586980312524470804191878e-02
            4.47295133944527745217811798283947e-02 9.10540973211094506467588871601038e-01 4.47295133944527467662055641994812e-02 2.55776756586980694163635519089439e-02
            9.10540973211094728512193796632346e-01 4.47295133944526704383726212199690e-02 4.47295133944526010494335821476852e-02 2.55776756586979653329549933005183e-02
            3.68384120547361887387793899506505e-02 2.21962989160765650220596967301390e-01 7.41198598784498230429562681820244e-01 4.32835393772892373931426845956594e-02
            3.68384120547361054720525430639100e-02 7.41198598784498008384957756788936e-01 2.21962989160765844509626276703784e-01 4.32835393772892026986731650595175e-02
            2.21962989160765872265201892332698e-01 3.68384120547363830278086993530451e-02 7.41198598784497786340352831757627e-01 4.32835393772894663766415135341958e-02
            2.21962989160765983287504354848352e-01 7.41198598784497786340352831757627e-01 3.68384120547362581277184290229343e-02 4.32835393772893345376573392968567e-02
            7.41198598784498008384957756788936e-01 3.68384120547365218056867774976126e-02 2.21962989160765455931567657898995e-01 4.32835393772895357655805526064796e-02
            7.41198598784498230429562681820244e-01 2.21962989160765400420416426641168e-01 3.68384120547363691500208915385883e-02 4.32835393772894455599598018125107e-02
            ];
        
    case  10
        
        % ALG. DEG.:   10
        % PTS CARD.:   25
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.1e-16
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333425851918718763045e-01 9.08179903827535384142066732238163e-02
            4.85577633383657003562916543160100e-01 4.85577633383657725207882549511851e-01 2.88447332326852712292009073280497e-02 3.67259577564667469284920287009299e-02
            4.85577633383657003562916543160100e-01 2.88447332326852573514130995135929e-02 4.85577633383657725207882549511851e-01 3.67259577564666914173407974431029e-02
            2.88447332326852330652844358382936e-02 4.85577633383657003562916543160100e-01 4.85577633383657780719033780769678e-01 3.67259577564666636617651818141894e-02
            1.09481575485037757600181862471800e-01 1.09481575485036522477066966985149e-01 7.81036849029925650533812131470768e-01 4.53210594355279022815707890003978e-02
            1.09481575485036189410159579438186e-01 7.81036849029925872578417056502076e-01 1.09481575485037896378059940616367e-01 4.53210594355279300371464046293113e-02
            7.81036849029926094623021981533384e-01 1.09481575485037480044425706182665e-01 1.09481575485036425332552312283951e-01 4.53210594355279577927220202582248e-02
            1.41707219414880319918736972795159e-01 3.07939838764120288328030028424109e-01 5.50352941820999363997657383151818e-01 7.27579168454201147264726046159922e-02
            1.41707219414879487251468503927754e-01 5.50352941820998697863842608057894e-01 3.07939838764121787129113272385439e-01 7.27579168454200037041701421003381e-02
            3.07939838764121620595659578611958e-01 1.41707219414879626029346582072321e-01 5.50352941820998697863842608057894e-01 7.27579168454200314597457577292516e-02
            3.07939838764120177305727565908455e-01 5.50352941820999475019959845667472e-01 1.41707219414880403185463819681900e-01 7.27579168454202118709872593171895e-02
            5.50352941820999363997657383151818e-01 1.41707219414880347674312588424073e-01 3.07939838764120288328030028424109e-01 7.27579168454201563598360280593624e-02
            5.50352941820998808886145070573548e-01 3.07939838764121565084508347354131e-01 1.41707219414879626029346582072321e-01 7.27579168454201147264726046159922e-02
            2.50035347626866161219183481989603e-02 2.46672560639902560541969478435931e-01 7.28323904597410809458324365550652e-01 2.83272425310576447810007039151969e-02
            2.50035347626861442771328825074306e-02 7.28323904597410698436021903034998e-01 2.46672560639903171164633022272028e-01 2.83272425310572562029420851104078e-02
            2.46672560639902394008515784662450e-01 2.50035347626861338687920266465881e-02 7.28323904597411475592139140644576e-01 2.83272425310572839585177007393213e-02
            2.46672560639902199719486475260055e-01 7.28323904597411142525231753097614e-01 2.50035347626866855108573872712441e-02 2.83272425310577141699397429874807e-02
            7.28323904597410365369114515488036e-01 2.50035347626865606107671169411333e-02 2.46672560639903060142330559756374e-01 2.83272425310576829449171754049530e-02
            7.28323904597411031502929290581960e-01 2.46672560639902782586574403467239e-01 2.50035347626861859104963059508009e-02 2.83272425310573360002219800435341e-02
            9.54081540029997135499151283966057e-03 6.68032510122005263042410661000758e-02 9.23655933587499533565789988642791e-01 9.42166696373300700806385776786556e-03
            9.54081540029878827358089665722218e-03 9.23655933587500643788814613799332e-01 6.68032510122005263042410661000758e-02 9.42166696373258373553571942693452e-03
            6.68032510121999989483043691507191e-02 9.54081540029910052380657248249918e-03 9.23655933587500976855722001346294e-01 9.42166696373265659392171045283249e-03
            6.68032510121999989483043691507191e-02 9.23655933587499866632697376189753e-01 9.54081540030016217457387028844096e-03 9.42166696373307292755594488653514e-03
            9.23655933587499533565789988642791e-01 9.54081540029977012706829953003762e-03 6.68032510122006928376947598735569e-02 9.42166696373296017053000639407401e-03
            9.23655933587500976855722001346294e-01 6.68032510121999989483043691507191e-02 9.54081540029902419597362950298702e-03 9.42166696373267394115647022090343e-03
            ];
        
    case  11
        
        % ALG. DEG.:   11
        % PTS CARD.:   27
        % NEG. W.  :    0
        % OUT PTS. :    3
        % M.E.INF. : 8.9e-16
        
        xyw_bar=[
            5.34611048270758226763632592337672e-01 5.34611048270758337785935054853326e-01 -6.92220965415165645495676471909974e-02 9.27006328960677241854404506682386e-04
            5.34611048270758559830539979884634e-01 -6.92220965415166339385066862632812e-02 5.34611048270758115741330129822018e-01 9.27006328960675723971363026976178e-04
            -6.92220965415166061829310706343676e-02 5.34611048270758337785935054853326e-01 5.34611048270758337785935054853326e-01 9.27006328960676591333101015379725e-04
            3.98969302965854977216508814308327e-01 3.98969302965855421305718664370943e-01 2.02061394068289601477772521320730e-01 7.71495349148130921124177916681219e-02
            3.98969302965854977216508814308327e-01 2.02061394068289573722196905691817e-01 3.98969302965855421305718664370943e-01 7.71495349148130921124177916681219e-02
            2.02061394068289518211045674433990e-01 3.98969302965854977216508814308327e-01 3.98969302965855532328021126886597e-01 7.71495349148131059902055994825787e-02
            2.03309900431282303268076816493704e-01 2.03309900431282192245774353978049e-01 5.93380199137435559997300060786074e-01 5.93229773807740154434320345444576e-02
            2.03309900431282303268076816493704e-01 5.93380199137435004885787748207804e-01 2.03309900431282719601711050927406e-01 5.93229773807740431990076501733711e-02
            5.93380199137434560796577898145188e-01 2.03309900431282719601711050927406e-01 2.03309900431282719601711050927406e-01 5.93229773807740987101588814311981e-02
            1.19350912282581225754007903105958e-01 1.19350912282581253509583518734871e-01 7.61298175434837576247559809416998e-01 3.61845405034179820602524557671131e-02
            1.19350912282581517187551867209550e-01 7.61298175434837243180652421870036e-01 1.19350912282581211876220095291501e-01 3.61845405034181000214488221899956e-02
            7.61298175434837132158349959354382e-01 1.19350912282581433920825020322809e-01 1.19350912282581433920825020322809e-01 3.61845405034181832881756690767361e-02
            3.23649481112758610756863220103696e-02 3.23649481112758610756863220103696e-02 9.35270103777448236215263932535890e-01 1.36597310026778488711851622383620e-02
            3.23649481112758957701558415465115e-02 9.35270103777448236215263932535890e-01 3.23649481112758818923680337320548e-02 1.36597310026778558100790661455903e-02
            9.35270103777448236215263932535890e-01 3.23649481112758957701558415465115e-02 3.23649481112758680145802259175980e-02 1.36597310026778696878668739600471e-02
            5.01781383104946410167457315765205e-02 3.56620648261292472191996694164118e-01 5.93201213428212859035681958630448e-01 5.23371119622040303909216163447127e-02
            5.01781383104946548945335393909772e-02 5.93201213428212636991077033599140e-01 3.56620648261292694236601619195426e-01 5.23371119622040512076033280663978e-02
            3.56620648261292583214299156679772e-01 5.01781383104946340778518276692921e-02 5.93201213428212748013379496114794e-01 5.23371119622040373298155202519411e-02
            3.56620648261292527703147925421945e-01 5.93201213428212859035681958630448e-01 5.01781383104945577500188846897800e-02 5.23371119622040859020728476025397e-02
            5.93201213428212636991077033599140e-01 5.01781383104947034667908667415759e-02 3.56620648261292638725450387937599e-01 5.23371119622040997798606554169965e-02
            5.93201213428212636991077033599140e-01 3.56620648261292638725450387937599e-01 5.01781383104947242834725784632610e-02 5.23371119622040859020728476025397e-02
            2.10220165361662721104085704837416e-02 1.71488980304041471747211744514061e-01 8.07489003159792217978463213512441e-01 2.07076596391406707320381741510573e-02
            2.10220165361663657854762732313247e-02 8.07489003159791995933858288481133e-01 1.71488980304041693791816669545369e-01 2.07076596391407366515302612697269e-02
            1.71488980304041555013938591400802e-01 2.10220165361663310910067536951829e-02 8.07489003159792106956160750996787e-01 2.07076596391406915487198858727425e-02
            1.71488980304041527258362975771888e-01 8.07489003159792217978463213512441e-01 2.10220165361661992520225794578437e-02 2.07076596391406395070156065685296e-02
            8.07489003159792106956160750996787e-01 2.10220165361663102743250419734977e-02 1.71488980304041582769514207029715e-01 2.07076596391406811403790300118999e-02
            8.07489003159792217978463213512441e-01 1.71488980304041555013938591400802e-01 2.10220165361662270075981950867572e-02 2.07076596391406846098259819655141e-02
            ];
        
    case  12
        
        % ALG. DEG.:   12
        % PTS CARD.:   33
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.9e-16
        
        xyw_bar=[
            4.88217389773805021668806602974655e-01 4.88217389773804744113050446685520e-01 2.35652204523902342181429503398249e-02 2.57310664404553114670282099041287e-02
            4.88217389773805188202260296748136e-01 2.35652204523902238098020944789823e-02 4.88217389773804633090747984169866e-01 2.57310664404553218753690657649713e-02
            2.35652204523902342181429503398249e-02 4.88217389773805021668806602974655e-01 4.88217389773804799624201677943347e-01 2.57310664404553357531568735794281e-02
            4.39724392294460586505522314837435e-01 4.39724392294459975882858771001338e-01 1.20551215411079437611618914161227e-01 4.36925445380384261384598687527614e-02
            4.39724392294460142416312464774819e-01 1.20551215411079423733831106346770e-01 4.39724392294460419972068621063954e-01 4.36925445380384261384598687527614e-02
            1.20551215411079465367194529790140e-01 4.39724392294460808550127239868743e-01 4.39724392294459698327102614712203e-01 4.36925445380383567495208296804776e-02
            2.71210385012115751113270789574017e-01 2.71210385012116195202480639636633e-01 4.57579229975768109195399802047177e-01 6.28582242178850614244822736509377e-02
            2.71210385012116028669026945863152e-01 4.57579229975768331240004727078485e-01 2.71210385012115640090968327058363e-01 6.28582242178850753022700814653945e-02
            4.57579229975767998173097339531523e-01 2.71210385012115862135573252089671e-01 2.71210385012116195202480639636633e-01 6.28582242178850475466944658364810e-02
            1.27576145541585861398203860517242e-01 1.27576145541586000176081938661810e-01 7.44847708916828166181289816449862e-01 3.47961129307089653273088458718121e-02
            1.27576145541585916909355091775069e-01 7.44847708916828055158987353934208e-01 1.27576145541586027931657554290723e-01 3.47961129307089236939454224284418e-02
            7.44847708916828388225894741481170e-01 1.27576145541585778131477013630501e-01 1.27576145541585833642628244888328e-01 3.47961129307088751216880950778432e-02
            2.13173504532104124420399671180348e-02 2.13173504532104228503808229788774e-02 9.57365299093579258382646912650671e-01 6.16626105155903929400107443825618e-03
            2.13173504532104124420399671180348e-02 9.57365299093579258382646912650671e-01 2.13173504532103708086765436746646e-02 6.16626105155900806897850685572848e-03
            9.57365299093579258382646912650671e-01 2.13173504532103569308887358602078e-02 2.13173504532103846864643514891213e-02 6.16626105155901067106372082093912e-03
            1.15343494534697874054351984796085e-01 2.75713269685514628104527901086840e-01 6.08943235779787483963332306302618e-01 4.03715577663809888075974185994710e-02
            1.15343494534698040587805678569566e-01 6.08943235779787928052542156365234e-01 2.75713269685514017481864357250743e-01 4.03715577663808292130376287332183e-02
            2.75713269685514017481864357250743e-01 1.15343494534698207121259372343047e-01 6.08943235779787817030239693849580e-01 4.03715577663809471742339951561007e-02
            2.75713269685514295037620513539878e-01 6.08943235779787928052542156365234e-01 1.15343494534697832420988561352715e-01 4.03715577663809402353400912488723e-02
            6.08943235779787928052542156365234e-01 1.15343494534697971198866639497282e-01 2.75713269685514128504166819766397e-01 4.03715577663809749298096107850142e-02
            6.08943235779787928052542156365234e-01 2.75713269685513906459561894735089e-01 1.15343494534698165487895948899677e-01 4.03715577663809332964461873416440e-02
            2.28383322222570142590036823548871e-02 2.81325580989939649789732811768772e-01 6.95836086787803287379006178525742e-01 2.23567732023034135280248335675424e-02
            2.28383322222570489534732018910290e-02 6.95836086787803620445913566072704e-01 2.81325580989939316722825424221810e-01 2.23567732023034690391760648253694e-02
            2.81325580989939871834337736800080e-01 2.28383322222570697701549136127142e-02 6.95836086787803065334401253494434e-01 2.23567732023034933253047285006687e-02
            2.81325580989939705300884043026599e-01 6.95836086787803287379006178525742e-01 2.28383322222569518089585471898317e-02 2.23567732023033996502370257530856e-02
            6.95836086787803620445913566072704e-01 2.28383322222570107895567304012729e-02 2.81325580989939372233976655479637e-01 2.23567732023034239363656894283849e-02
            6.95836086787803509423611103557050e-01 2.81325580989939427745127886737464e-01 2.28383322222570628312610097054858e-02 2.23567732023034725086230167789836e-02
            2.57340505483302020572899948547274e-02 1.16251915907597244692617266537127e-01 8.58014033544072574066774450329831e-01 1.73162311086588818764830932650511e-02
            2.57340505483302159350778026691842e-02 8.58014033544072796111379375361139e-01 1.16251915907596981014648918062449e-01 1.73162311086588784070361413114369e-02
            1.16251915907597300203768497794954e-01 2.57340505483302367517595143908693e-02 8.58014033544072463044471987814177e-01 1.73162311086589165709526128011930e-02
            1.16251915907597286325980689980497e-01 8.58014033544072463044471987814177e-01 2.57340505483302228739717065764125e-02 1.73162311086589165709526128011930e-02
            8.58014033544072796111379375361139e-01 2.57340505483302089961838987619558e-02 1.16251915907596994892436725876905e-01 1.73162311086588714681422374042086e-02
            8.58014033544072685089076912845485e-01 1.16251915907597078159163572763646e-01 2.57340505483302367517595143908693e-02 1.73162311086588541209074776361376e-02
            ];
        
    case  13
        
        % ALG. DEG.:   13
        % PTS CARD.:   37
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.8e-16
        
        xyw_bar=[
            3.33333333333333592385372412536526e-01 3.33333333333334036474582262599142e-01 3.33333333333332371140045324864332e-01 5.25209234008021941253474551558611e-02
            4.95048184939703816986877882300178e-01 4.95048184939705426810263588777161e-01 9.90363012059075620285852892266121e-03 1.12801452093294844042370428383038e-02
            4.95048184939704982721053738714545e-01 9.90363012059061048608654687086528e-03 4.95048184939704372098390194878448e-01 1.12801452093294826695135668614967e-02
            9.90363012059072150838900938651932e-03 4.95048184939704982721053738714545e-01 4.95048184939704316587238963620621e-01 1.12801452093295364459413221425166e-02
            4.68716635109574675244203945112531e-01 4.68716635109573842576935476245126e-01 6.25667297808515376900118099001702e-02 3.14235183624539740621095518235961e-02
            4.68716635109573787065784244987299e-01 6.25667297808519817792216599627864e-02 4.68716635109574286666145326307742e-01 3.14235183624543418234864589067001e-02
            6.25667297808519123902826208905026e-02 4.68716635109573731554633013729472e-01 4.68716635109574397688447788823396e-01 3.14235183624541614122449573187623e-02
            4.14521336801277617833960675852722e-01 4.14521336801276951700145900758798e-01 1.70957326397445374954742192130652e-01 4.70725025041943342940520267347893e-02
            4.14521336801276618633238513211836e-01 1.70957326397446957022552282978722e-01 4.14521336801276452099784819438355e-01 4.70725025041945355219752400444122e-02
            1.70957326397446429666615586029366e-01 4.14521336801277062722448363274452e-01 4.14521336801276507610936050696182e-01 4.70725025041944522552483931576717e-02
            2.29399572042831012552710490126628e-01 2.29399572042831734197676496478380e-01 5.41200855914337308760764244652819e-01 4.73635865363547237105734666329226e-02
            2.29399572042831512153071571447072e-01 5.41200855914337974894579019746743e-01 2.29399572042830568463500640064012e-01 4.73635865363547653439368900762929e-02
            5.41200855914337974894579019746743e-01 2.29399572042831456641920340189245e-01 2.29399572042830568463500640064012e-01 4.73635865363547028938917549112375e-02
            1.14424495196330244795035468996502e-01 1.14424495196330078261581775223021e-01 7.71151009607339732454533987038303e-01 3.11675290457938437838070200314178e-02
            1.14424495196330369695125739326613e-01 7.71151009607339288365324136975687e-01 1.14424495196330355817337931512157e-01 3.11675290457939339894277708253867e-02
            7.71151009607339732454533987038303e-01 1.14424495196329759072462195490516e-01 1.14424495196330508473003817471181e-01 3.11675290457938507227009239386462e-02
            2.48113913634589455803336477401899e-02 2.48113913634589906831440231371744e-02 9.50377217273082108839332704519620e-01 7.97577146507436399169854723822937e-03
            2.48113913634589559886745036010325e-02 9.50377217273082108839332704519620e-01 2.48113913634588900691824164823629e-02 7.97577146507434837918726344696552e-03
            9.50377217273081997817030242003966e-01 2.48113913634590253776135426733163e-02 2.48113913634589768053562153227176e-02 7.97577146507437093059245114545774e-03
            9.48538283795782588558864745209576e-02 2.68794997058761031549778408589191e-01 6.36351174561660792861061963776592e-01 3.68484027287322435384453456208576e-02
            9.48538283795784809004913995522656e-02 6.36351174561660126727247188682668e-01 2.68794997058761420127837027393980e-01 3.68484027287322574162331534353143e-02
            2.68794997058761031549778408589191e-01 9.48538283795787862118231714703143e-02 6.36351174561660126727247188682668e-01 3.68484027287322643551270573425427e-02
            2.68794997058761697683593183683115e-01 6.36351174561660348771852113713976e-01 9.48538283795778980334034713450819e-02 3.68484027287321463939306909196603e-02
            6.36351174561659793660339801135706e-01 9.48538283795786613117329011402035e-02 2.68794997058761531150139489909634e-01 3.68484027287321949661880182702589e-02
            6.36351174561661347972574276354862e-01 2.68794997058760809505173483557883e-01 9.48538283795778425222522400872549e-02 3.68484027287321810884002104558022e-02
            1.81007732788068115281632231017284e-02 2.91730066734287862839636318312841e-01 6.90169159986905311754412650770973e-01 1.74014633038219561356196862789147e-02
            1.81007732788068878559961660812405e-02 6.90169159986905533799017575802281e-01 2.91730066734287585283880162023706e-01 1.74014633038220428717934851192695e-02
            2.91730066734287696306182624539360e-01 1.81007732788069572449352051535243e-02 6.90169159986905311754412650770973e-01 1.74014633038221157301794761451674e-02
            2.91730066734287363239275236992398e-01 6.90169159986905866865924963349244e-01 1.81007732788067698947997996583581e-02 1.74014633038219075633623589283161e-02
            6.90169159986905533799017575802281e-01 1.81007732788069919394047246896662e-02 2.91730066734287474261577699508052e-01 1.74014633038221191996264280987816e-02
            6.90169159986905200732110188255319e-01 2.91730066734288251417694937117631e-01 1.81007732788065478501948746270500e-02 1.74014633038218381744233198560323e-02
            2.22330766740901838995903005979926e-02 1.26357385491668749999405463313451e-01 8.51409537834241114673261563439155e-01 1.55217868390449525300978095287974e-02
            2.22330766740901596134616369226933e-02 8.51409537834241225695564025954809e-01 1.26357385491668638977103000797797e-01 1.55217868390449646731621413664470e-02
            1.26357385491668638977103000797797e-01 2.22330766740900936939695498040237e-02 8.51409537834241225695564025954809e-01 1.55217868390449403870334776911477e-02
            1.26357385491668527954800538282143e-01 8.51409537834241336717866488470463e-01 2.22330766740901353273329732473940e-02 1.55217868390449438564804296447619e-02
            8.51409537834241447740168950986117e-01 2.22330766740900208355835587781257e-02 1.26357385491668527954800538282143e-01 1.55217868390448831411587704565136e-02
            8.51409537834240892628656638407847e-01 1.26357385491668916532859157086932e-01 2.22330766740901908384842045052210e-02 1.55217868390449958981847089489747e-02
            ];
        
    case  14
        
        % ALG. DEG.:   14
        % PTS CARD.:   42
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 8.5e-16
        
        xyw_bar=[
            4.88963910362178899582374924648320e-01 4.88963910362178399982013843327877e-01 2.20721792756427004356112320238026e-02 2.18835813694288824904354839873122e-02
            4.88963910362178344470862612070050e-01 2.20721792756427559467624632816296e-02 4.88963910362178899582374924648320e-01 2.18835813694288963682232918017689e-02
            2.20721792756427316606337996063303e-02 4.88963910362178955093526155906147e-01 4.88963910362178288959711380812223e-01 2.18835813694288790209885320336980e-02
            4.17644719340453995926054631127045e-01 4.17644719340453995926054631127045e-01 1.64710561319092008147890737745911e-01 3.27883535441253204378853070011246e-02
            4.17644719340453995926054631127045e-01 1.64710561319092452237100587808527e-01 4.17644719340453551836844781064428e-01 3.27883535441252926823096913722111e-02
            1.64710561319092008147890737745911e-01 4.17644719340453995926054631127045e-01 4.17644719340453995926054631127045e-01 3.27883535441253898268243460734084e-02
            2.73477528308838646609046918456443e-01 2.73477528308838702120198149714270e-01 4.53044943382322651270754931829288e-01 5.17741045072915503921784363683400e-02
            2.73477528308838646609046918456443e-01 4.53044943382322706781906163087115e-01 2.73477528308838646609046918456443e-01 5.17741045072915989644357637189387e-02
            4.53044943382322595759603700571461e-01 2.73477528308838813142500612229924e-01 2.73477528308838591097895687198616e-01 5.17741045072915156977089168321982e-02
            1.77205532412543192988252371833369e-01 1.77205532412543720344189068782725e-01 6.45588935174913114423134175012819e-01 4.21625887369930021919550711118063e-02
            1.77205532412543526055159759380331e-01 6.45588935174913225445436637528474e-01 1.77205532412543220743827987462282e-01 4.21625887369929952530611672045779e-02
            6.45588935174913114423134175012819e-01 1.77205532412543304010554834349023e-01 1.77205532412543581566310990638158e-01 4.21625887369930160697428789262631e-02
            6.17998830908723581489816467637866e-02 6.17998830908729618327512866926554e-02 8.76400233818254714712736586079700e-01 1.44336996697766832553622151635864e-02
            6.17998830908726218269499952384649e-02 8.76400233818255158801946436142316e-01 6.17998830908722540655730881553609e-02 1.44336996697766416219987917202161e-02
            8.76400233818254492668131661048392e-01 6.17998830908724830490719170938974e-02 6.17998830908730242827964218577108e-02 1.44336996697766953984265470012360e-02
            1.93909612487010336867587767528676e-02 1.93909612487010336867587767528676e-02 9.61218077502598022832103197288234e-01 4.92340360240008274861356341034480e-03
            1.93909612487010267478648728456392e-02 9.61218077502597911809800734772580e-01 1.93909612487010996062508638715371e-02 4.92340360240005759512316174664193e-03
            9.61218077502597911809800734772580e-01 1.93909612487010683812282962890094e-02 1.93909612487010198089709689384108e-02 4.92340360240009315695441927118736e-03
            5.71247574036476968295517053775256e-02 1.72266687821355873877493536383554e-01 7.70608554774996346026227911352180e-01 2.46657532125636319575434640682943e-02
            5.71247574036479952019895733883459e-02 7.70608554774996790115437761414796e-01 1.72266687821355235499254376918543e-01 2.46657532125636770603538394652787e-02
            1.72266687821355485299434917578765e-01 5.71247574036482380632762101413391e-02 7.70608554774996346026227911352180e-01 2.46657532125636909381416472797355e-02
            1.72266687821355596321737380094419e-01 7.70608554774996568070832836383488e-01 5.71247574036478633630053991510067e-02 2.46657532125636701214599355580503e-02
            7.70608554774996123981622986320872e-01 5.71247574036478772407932069654635e-02 1.72266687821356012655371614528121e-01 2.46657532125636493047782238363652e-02
            7.70608554774996457048530373867834e-01 1.72266687821355513055010533207678e-01 5.71247574036480298964590929244878e-02 2.46657532125636597131190796972078e-02
            9.29162493569716391172974567780329e-02 3.36861459796345241723969365921221e-01 5.70222290846683188547672216373030e-01 3.85715107870606213613662305306207e-02
            9.29162493569718472841145739948843e-02 5.70222290846683521614579603919992e-01 3.36861459796344631101305822085124e-01 3.85715107870606768725174617884477e-02
            3.36861459796344908657061978374259e-01 9.29162493569720276953560755828221e-02 5.70222290846683077525369753857376e-01 3.85715107870607532003504047679598e-02
            3.36861459796345075190515672147740e-01 5.70222290846683188547672216373030e-01 9.29162493569717362618121114792302e-02 3.85715107870606629947296539739909e-02
            5.70222290846682966503067291341722e-01 9.29162493569718056507511505515140e-02 3.36861459796345241723969365921221e-01 3.85715107870606491169418461595342e-02
            5.70222290846683188547672216373030e-01 3.36861459796344853145910747116432e-01 9.29162493569719583064170365105383e-02 3.85715107870606699336235578812193e-02
            1.46469500556543723301938797476396e-02 2.98372882136258066321232718109968e-01 6.86980167808087571756914258003235e-01 1.44363081135337814608909923208557e-02
            1.46469500556544174330042551446240e-02 6.86980167808088126868426570581505e-01 2.98372882136257455698569174273871e-01 1.44363081135338473803830794395253e-02
            2.98372882136257511209720405531698e-01 1.46469500556545093733484819154000e-02 6.86980167808088015846124108065851e-01 1.44363081135338820748525989756672e-02
            2.98372882136258010810081486852141e-01 6.86980167808087571756914258003235e-01 1.46469500556544174330042551446240e-02 1.44363081135338352373187476018757e-02
            6.86980167808087571756914258003235e-01 1.46469500556544347802390149126950e-02 2.98372882136258010810081486852141e-01 1.44363081135338248289778917410331e-02
            6.86980167808088015846124108065851e-01 2.98372882136257511209720405531698e-01 1.46469500556544729441554864024511e-02 1.44363081135338265637013677178402e-02
            1.26833093287180413179238147591832e-03 1.18974497696956907355492205624614e-01 8.79757171370171286994832371419761e-01 5.01022883850060687244143409202479e-03
            1.26833093287203485001468639126188e-03 8.79757171370171286994832371419761e-01 1.18974497696956671433099472778849e-01 5.01022883850066932248656925708019e-03
            1.18974497696956782455401935294503e-01 1.26833093287230134690868332825175e-03 8.79757171370170953927924983872799e-01 5.01022883850074478295777424818880e-03
            1.18974497696956935111067821253528e-01 8.79757171370171175972529908904107e-01 1.26833093287193054976569328573532e-03 5.01022883850061381133533799925317e-03
            8.79757171370170953927924983872799e-01 1.26833093287197239996955122620648e-03 1.18974497696957073888945899398095e-01 5.01022883850065024052833351220215e-03
            8.79757171370171175972529908904107e-01 1.18974497696956671433099472778849e-01 1.26833093287215259437061831704341e-03 5.01022883850072223155258654969657e-03
            ];
        
    case  15
        
        % ALG. DEG.:   15
        % PTS CARD.:   48
        % NEG. W.  :    0
        % OUT PTS. :    9
        % M.E.INF. : 8.5e-16
        
        xyw_bar=[
            5.06972916858242972359960276662605e-01 5.06972916858242972359960276662605e-01 -1.39458337164859447199205533252098e-02 1.91687564284861848042107457246175e-03
            5.06972916858242972359960276662605e-01 -1.39458337164859204337918896499104e-02 5.06972916858242972359960276662605e-01 1.91687564284862151618715753187416e-03
            -1.39458337164858666573641343688905e-02 5.06972916858242972359960276662605e-01 5.06972916858242972359960276662605e-01 1.91687564284862303407019901158037e-03
            4.31406354283023041329414581923629e-01 4.31406354283022208662146113056224e-01 1.37187291433954694497288073762320e-01 4.42490272711447210629565063300106e-02
            4.31406354283022763773658425634494e-01 1.37187291433954777764014920649061e-01 4.31406354283022430706751038087532e-01 4.42490272711447349407443141444674e-02
            1.37187291433954750008439305020147e-01 4.31406354283023207862868275697110e-01 4.31406354283022097639843650540570e-01 4.42490272711447210629565063300106e-02
            2.77693644847144427423302204260835e-01 2.77693644847144538445604666776489e-01 4.44612710305711034131093128962675e-01 5.11865487188521220418557788889302e-02
            2.77693644847144482934453435518662e-01 4.44612710305711256175698053993983e-01 2.77693644847144205378697279229527e-01 5.11865487188521081640679710744735e-02
            4.44612710305711311686849285251810e-01 2.77693644847144316400999741745181e-01 2.77693644847144316400999741745181e-01 5.11865487188521151029618749817018e-02
            1.26464891041253996162296857619367e-01 1.26464891041254440251506707681983e-01 7.47070217917491508075045203440823e-01 2.36877358706878600458356487479250e-02
            1.26464891041254023917872473248281e-01 7.47070217917492507275767366081709e-01 1.26464891041253468806360160670010e-01 2.36877358706877941263435616292554e-02
            7.47070217917492951364977216144325e-01 1.26464891041253385539633313783270e-01 1.26464891041253663095389470072405e-01 2.36877358706877871874496577220270e-02
            7.08083859746862082085172573897580e-02 7.08083859746859722861245245439932e-02 8.58383228050627833383146025880706e-01 1.32897756900206120284080313354025e-02
            7.08083859746855559524902901102905e-02 8.58383228050628277472355875943322e-01 7.08083859746861943307294495753013e-02 1.32897756900204836588708090516775e-02
            8.58383228050628610539263263490284e-01 7.08083859746857918748830229560554e-02 7.08083859746855975858537135536608e-02 1.32897756900204194741021979098150e-02
            1.89651702410733209724913450600070e-02 1.89651702410733452586200087353063e-02 9.62069659517853281727184366900474e-01 4.74891660819185684738741315413790e-03
            1.89651702410733730141956243642198e-02 9.62069659517853281727184366900474e-01 1.89651702410733591364078165497631e-02 4.74891660819184123487612936287405e-03
            9.62069659517853392749486829416128e-01 1.89651702410732689307870657557942e-02 1.89651702410733383197261048280780e-02 4.74891660819183429598222545564568e-03
            1.33734161966621162287438551175001e-01 2.61311371140088066766793417627923e-01 6.04954466893290798701343646825990e-01 3.85500725995925053180002350927680e-02
            1.33734161966621106776287319917174e-01 6.04954466893292019946670734498184e-01 2.61311371140086845521466329955729e-01 3.85500725995925122568941389999964e-02
            2.61311371140087567166432336307480e-01 1.33734161966621190043014166803914e-01 6.04954466893291131768251034372952e-01 3.85500725995925261346819468144531e-02
            2.61311371140087789211037261338788e-01 6.04954466893291242790553496888606e-01 1.33734161966620912487258010514779e-01 3.85500725995925053180002350927680e-02
            6.04954466893291686879763346951222e-01 1.33734161966620995753984857401520e-01 2.61311371140087289610676180018345e-01 3.85500725995925191957880429072247e-02
            6.04954466893292019946670734498184e-01 2.61311371140087012054920023729210e-01 1.33734161966620967998409241772606e-01 3.85500725995925122568941389999964e-02
            3.63666773969168327651502181652177e-02 5.75586555512814679325117594999028e-01 3.88046767090268529543095610279124e-01 2.72158143206242680856021820545720e-02
            3.63666773969168813374075455158163e-02 3.88046767090269084654607922857394e-01 5.75586555512814013191302819905104e-01 2.72158143206242611467082781473437e-02
            5.75586555512814346258210207452066e-01 3.63666773969168188873624103507609e-02 3.88046767090268862610002997826086e-01 2.72158143206242646161552301009579e-02
            5.75586555512814568302815132483374e-01 3.88046767090268585054246841536951e-01 3.63666773969168466429380259796744e-02 2.72158143206242472689204703328869e-02
            3.88046767090268751587700535310432e-01 3.63666773969168188873624103507609e-02 5.75586555512814457280512669967720e-01 2.72158143206242750244960859618004e-02
            3.88046767090269029143456691599567e-01 5.75586555512814124213605282420758e-01 3.63666773969169021540892572375014e-02 2.72158143206242611467082781473437e-02
            -1.01748831265706645571311739217890e-02 2.85712220049916032493797501956578e-01 7.24462663076654633798057147942018e-01 2.18207736679703430057730706437269e-03
            -1.01748831265708397642022475793055e-02 7.24462663076654744820359610457672e-01 2.85712220049916143516099964472232e-01 2.18207736679700307555473948184499e-03
            2.85712220049915976982646270698751e-01 -1.01748831265706662918546498985961e-02 7.24462663076654633798057147942018e-01 2.18207736679703516793904505277624e-03
            2.85712220049915976982646270698751e-01 7.24462663076654522775754685426364e-01 -1.01748831265705552695521873829421e-02 2.18207736679705251517380482084718e-03
            7.24462663076654633798057147942018e-01 -1.01748831265706385362790342696826e-02 2.85712220049916032493797501956578e-01 2.18207736679702953008774812815318e-03
            7.24462663076654744820359610457672e-01 2.85712220049915976982646270698751e-01 -1.01748831265707218030058811564231e-02 2.18207736679701391757646433688933e-03
            3.68438698758782462450156458544370e-02 2.15599664072284286087111127017124e-01 7.47556466051837453790085419313982e-01 2.15053198477313869951821345694043e-02
            3.68438698758781282838192794315546e-02 7.47556466051838008901597731892252e-01 2.15599664072283814242325661325594e-01 2.15053198477313731173943267549475e-02
            2.15599664072284008531354970727989e-01 3.68438698758782740005912614833505e-02 7.47556466051837675834690344345290e-01 2.15053198477314112813107982447036e-02
            2.15599664072284230575959895759297e-01 7.47556466051837564812387881829636e-01 3.68438698758781768560766067821532e-02 2.15053198477313349534778552651915e-02
            7.47556466051837786856992806860944e-01 3.68438698758781282838192794315546e-02 2.15599664072284091798081817614730e-01 2.15053198477313314840309033115773e-02
            7.47556466051837897879295269376598e-01 2.15599664072283925264628123841248e-01 3.68438698758781768560766067821532e-02 2.15053198477313349534778552651915e-02
            1.24598093311987720421818437444017e-02 1.03575616576385864942011494349572e-01 8.83964574092415356076912757998798e-01 7.67394263104869637365235135462171e-03
            1.24598093311986072434516259477277e-02 8.83964574092415800166122608061414e-01 1.03575616576385587386255338060437e-01 7.67394263104861223956376647947764e-03
            1.03575616576385753919709031833918e-01 1.24598093311986766323906650200115e-02 8.83964574092415578121517683030106e-01 7.67394263104865474028892791125145e-03
            1.03575616576385823308648070906202e-01 8.83964574092415356076912757998798e-01 1.24598093311988344922269789094571e-02 7.67394263104872586395144296034232e-03
            8.83964574092415467099215220514452e-01 1.24598093311987546949470839763308e-02 1.03575616576385781675284647462831e-01 7.67394263104868770003497147058624e-03
            8.83964574092415578121517683030106e-01 1.03575616576385656775194377132721e-01 1.24598093311987651032879398371733e-02 7.67394263104864780139502400402307e-03
            ];
        
    case  16
        
        % ALG. DEG.:   16
        % PTS CARD.:   52
        % NEG. W.  :    0
        % OUT PTS. :    6
        % M.E.INF. : 9.0e-16
        
        xyw_bar=[
            3.33333333333333259318465024989564e-01 3.33333333333333370340767487505218e-01 3.33333333333333370340767487505218e-01 4.68756974276416105151810143070179e-02
            4.97380541948438004151711311351391e-01 4.97380541948438781307828548960970e-01 5.23891610312315902930890842981171e-03 6.40587857858495018231748119319491e-03
            4.97380541948438004151711311351391e-01 5.23891610312323102033316146730613e-03 4.97380541948438725796677317703143e-01 6.40587857858496752955224096126585e-03
            5.23891610312315555986195647619752e-03 4.97380541948438004151711311351391e-01 4.97380541948438892330131011476624e-01 6.40587857858495104967921918159846e-03
            4.13469438549352197309616485654260e-01 4.13469438549352474865372641943395e-01 1.73061122901295272313859641144518e-01 4.17102967393868581669202910688909e-02
            4.13469438549352252820767716912087e-01 1.73061122901295300069435256773431e-01 4.13469438549352474865372641943395e-01 4.17102967393868304113446754399774e-02
            1.73061122901295105780405947371037e-01 4.13469438549352363843070179427741e-01 4.13469438549352585887675104459049e-01 4.17102967393868373502385793472058e-02
            4.70458599066991278725424763251795e-01 4.70458599066991500770029688283103e-01 5.90828018660172205045455484651029e-02 2.68914842500644034328782794318613e-02
            4.70458599066991112191971069478313e-01 5.90828018660173107101662992590718e-02 4.70458599066991500770029688283103e-01 2.68914842500644103717721833390897e-02
            5.90828018660172274434394523723313e-02 4.70458599066991112191971069478313e-01 4.70458599066991667303483382056584e-01 2.68914842500643722078557118493336e-02
            2.40553749969520863638550167706853e-01 2.40553749969520919149701398964680e-01 5.18892500060958217211748433328466e-01 4.21325227616496100768728183538769e-02
            2.40553749969520752616247705191199e-01 5.18892500060958328234050895844121e-01 2.40553749969520946905277014593594e-01 4.21325227616496378324484339827904e-02
            5.18892500060958106189445970812812e-01 2.40553749969520974660852630222507e-01 2.40553749969520919149701398964680e-01 4.21325227616496308935545300755621e-02
            1.47965794222572888383382405663724e-01 1.47965794222572777361079943148070e-01 7.04068411554854334255537651188206e-01 3.00002668427729463074271620826039e-02
            1.47965794222572666338777480632416e-01 7.04068411554854223233235188672552e-01 1.47965794222573054916836099437205e-01 3.00002668427729705935558257579032e-02
            7.04068411554854112210932726156898e-01 1.47965794222572832872231174405897e-01 1.47965794222573054916836099437205e-01 3.00002668427729775324497296651316e-02
            7.54651876574742169756504495126137e-02 7.54651876574742169756504495126137e-02 8.49069624685051538293123485345859e-01 1.42000989250241557715348506008013e-02
            7.54651876574740365644089479246759e-02 8.49069624685051649315425947861513e-01 7.54651876574743418757407198427245e-02 1.42000989250241800576635142761006e-02
            8.49069624685051316248518560314551e-01 7.54651876574743002423772963993542e-02 7.54651876574743835091041432860948e-02 1.42000989250241922007278461137503e-02
            1.65964026230248709592451916705613e-02 1.65964026230249403481842307428451e-02 9.66807194753950160936994961957680e-01 3.58246235127335438744311524317254e-03
            1.65964026230249230009494709747742e-02 9.66807194753950160936994961957680e-01 1.65964026230248640203512877633329e-02 3.58246235127336089265615015619915e-03
            9.66807194753950160936994961957680e-01 1.65964026230249715732067983253728e-02 1.65964026230248674897982397169471e-02 3.58246235127336349474136412140979e-03
            1.03575692245251996248533998823405e-01 2.96555596579887281727394565677969e-01 5.99868711174860735901859243313083e-01 3.27731474606273720517890524206450e-02
            1.03575692245251912981807151936664e-01 5.99868711174860402834951855766121e-01 2.96555596579887725816604415740585e-01 3.27731474606273859295768602351018e-02
            2.96555596579887337238545796935796e-01 1.03575692245252051759685230081232e-01 5.99868711174860513857254318281775e-01 3.27731474606273859295768602351018e-02
            2.96555596579887337238545796935796e-01 5.99868711174860624879556780797429e-01 1.03575692245251982370746191008948e-01 3.27731474606273998073646680495585e-02
            5.99868711174860291812649393250467e-01 1.03575692245252024004109614452318e-01 2.96555596579887670305453184482758e-01 3.27731474606273998073646680495585e-02
            5.99868711174860513857254318281775e-01 2.96555596579887503771999490709277e-01 1.03575692245251982370746191008948e-01 3.27731474606274136851524758640153e-02
            2.00834116554158628731308056103444e-02 3.37723063403078782584998407401145e-01 6.42193524941505389236340306524653e-01 1.52983062484411558995534008431605e-02
            2.00834116554158455258960458422735e-02 6.42193524941504723102525531430729e-01 3.37723063403079448718813182495069e-01 1.52983062484411645731707807271960e-02
            3.37723063403078782584998407401145e-01 2.00834116554158732814716614711870e-02 6.42193524941505389236340306524653e-01 1.52983062484411715120646846344243e-02
            3.37723063403078893607300869916799e-01 6.42193524941505167191735381493345e-01 2.00834116554159392009637485898566e-02 1.52983062484411853898524924488811e-02
            6.42193524941504834124827993946383e-01 2.00834116554158455258960458422735e-02 3.37723063403079337696510719979415e-01 1.52983062484411767162351125648456e-02
            6.42193524941504834124827993946383e-01 3.37723063403079337696510719979415e-01 2.00834116554158281786612860742025e-02 1.52983062484411801856820645184598e-02
            -4.34100261413866445742160138365762e-03 2.04748281642811641445334203126549e-01 7.99592720971326942347445765335578e-01 2.38624419283870294397620348547662e-03
            -4.34100261413873211163716447913430e-03 7.99592720971326609280538377788616e-01 2.04748281642812224312422131333733e-01 2.38624419283869210195447863043228e-03
            2.04748281642811696956485434384376e-01 -4.34100261413874251997802033997687e-03 7.99592720971327053369748227851233e-01 2.38624419283868256097536075799326e-03
            2.04748281642811946756665975044598e-01 7.99592720971326609280538377788616e-01 -4.34100261413850052605312157538719e-03 2.38624419283873677108398503321496e-03
            7.99592720971326831325143302819924e-01 -4.34100261413875466304235217762653e-03 2.04748281642811919001090359415684e-01 2.38624419283868256097536075799326e-03
            7.99592720971326609280538377788616e-01 2.04748281642811974512241590673511e-01 -4.34100261413858379277996846212773e-03 2.38624419283872289329617721875820e-03
            4.19417864680099103380506164739927e-02 1.89358492130622363402991936709441e-01 7.68699721401367663808912311651511e-01 1.90847927558988605944989558338420e-02
            4.19417864680098131935359617727954e-02 7.68699721401367441764307386620203e-01 1.89358492130622724225474939885316e-01 1.90847927558988814111806675555272e-02
            1.89358492130622363402991936709441e-01 4.19417864680098478880054813089373e-02 7.68699721401367774831214774167165e-01 1.90847927558988779417337156019130e-02
            1.89358492130622280136265089822700e-01 7.68699721401367663808912311651511e-01 4.19417864680100560548225985257886e-02 1.90847927558988883500745714627556e-02
            7.68699721401367552786609849135857e-01 4.19417864680098270713237695872522e-02 1.89358492130622613203172477369662e-01 1.90847927558989091667562831844407e-02
            7.68699721401367330742004924104549e-01 1.89358492130622613203172477369662e-01 4.19417864680100560548225985257886e-02 1.90847927558989091667562831844407e-02
            1.43173202306813591305756361293788e-02 8.52836156826571323286145798192592e-02 9.00399064086661482519957644399256e-01 6.85005454654197872693677240363286e-03
            1.43173202306813331097234964772724e-02 9.00399064086661371497655181883601e-01 8.52836156826572988620682735927403e-02 6.85005454654197525748982045001867e-03
            8.52836156826570351840999251180619e-02 1.43173202306814285195146752016626e-02 9.00399064086661482519957644399256e-01 6.85005454654199694153327016010735e-03
            8.52836156826571739619780032626295e-02 9.00399064086661482519957644399256e-01 1.43173202306813296402765445236582e-02 6.85005454654199347208631820649316e-03
            9.00399064086661260475352719367947e-01 1.43173202306814458667494349697336e-02 8.52836156826572988620682735927403e-02 6.85005454654201342140629193977475e-03
            9.00399064086661371497655181883601e-01 8.52836156826572572287048501493700e-02 1.43173202306813712736399679670285e-02 6.85005454654200214570369809052863e-03
            ];
        
    case  17
        
        % ALG. DEG.:   17
        % PTS CARD.:   61
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.6e-16
        
        xyw_bar=[
            3.33333333333340642301578782280558e-01 3.33333333333348802440809777181130e-01 3.33333333333310610768762671796139e-01 3.34371992908075454931093872801284e-02
            4.97170540556762186046313445331180e-01 4.97170540556774009921525703248335e-01 5.65891888646385954331208267831244e-03 5.09341544051346208771269274961924e-03
            4.97170540556774009921525703248335e-01 5.65891888646769675164094337560527e-03 4.97170540556758300265727257283288e-01 5.09341544051464777120852289726827e-03
            5.65891888645994600715027900150744e-03 4.97170540556774009921525703248335e-01 4.97170540556766016315748402121244e-01 5.09341544051025024719692169128393e-03
            4.82176322624617181844541846658103e-01 4.82176322624609909883730551882763e-01 3.56473547507729082717276014591334e-02 1.46708645276366709403470522943280e-02
            4.82176322624613629130863046157174e-01 3.56473547507641097542574470935506e-02 4.82176322624622288870455122378189e-01 1.46708645276301900134408029430233e-02
            3.56473547507499821662690919765737e-02 4.82176322624618181045264009298990e-01 4.82176322624631836788466898724437e-01 1.46708645276308197180625825239986e-02
            4.50239969020782893061038976156851e-01 4.50239969020782004882619276031619e-01 9.95200619584351020563417478115298e-02 2.43508783536622516741054766953312e-02
            4.50239969020783448172551288735121e-01 9.95200619584295093078551985854574e-02 4.50239969020787056397381320493878e-01 2.43508783536696450655600898471675e-02
            9.95200619584220708135902100366366e-02 4.50239969020793828757831533948774e-01 4.50239969020784058795214832571219e-01 2.43508783536720008200404663512018e-02
            4.00266239377405741439730491038063e-01 4.00266239377414734246229954806040e-01 1.99467521245179579825190785413724e-01 3.11075508689640303772172558183229e-02
            4.00266239377403465482530009467155e-01 1.99467521245205864355298785994819e-01 4.00266239377390697917746820166940e-01 3.11075508689749972990323811927738e-02
            1.99467521245202922264283529329987e-01 4.00266239377414512201625029774732e-01 4.00266239377382537778515825266368e-01 3.11075508689766244696528474378283e-02
            2.52141267970952986932076100856648e-01 2.52141267970952986932076100856648e-01 4.95717464058094081646999029544531e-01 3.12571112186287719536537110798236e-02
            2.52141267970967974942908540469944e-01 4.95717464058113954639139819846605e-01 2.52141267970918014906800408425624e-01 3.12571112186228738938353899357026e-02
            4.95717464058097578849526598787634e-01 2.52141267970972249301553347322624e-01 2.52141267970930171848920053889742e-01 3.12571112186226726659121766260796e-02
            1.62047004658467025395296445822169e-01 1.62047004658467747040262452173920e-01 6.75905990683065227564441102003912e-01 2.48156543396653092581338739819330e-02
            1.62047004658484705696963601440075e-01 6.75905990683077995129224291304126e-01 1.62047004658437243662660875997972e-01 2.48156543396644904686532129289844e-02
            6.75905990683063118140694314206485e-01 1.62047004658487370232222701815772e-01 1.62047004658449511627082983977743e-01 2.48156543396644418963958855783858e-02
            7.58758822607437238838556936570967e-02 7.58758822607484978428615818302205e-02 8.48248235478507806028858340141596e-01 1.40560730705570607196319699028209e-02
            7.58758822607369376456176723877434e-02 8.48248235478509693408000202907715e-01 7.58758822607533689463821247045416e-02 1.40560730705568837778374202684972e-02
            8.48248235478513801233191315986915e-01 7.58758822607351196554148486939084e-02 7.58758822607510791113938353191770e-02 1.40560730705562176440226451745730e-02
            1.56547269678219622135451061240019e-02 1.56547269678213585297754661951330e-02 9.68690546064356672317785523773637e-01 3.19467617377875703552070341117997e-03
            1.56547269678226734501702566149106e-02 9.68690546064356228228575673711020e-01 1.56547269678211087295949255349115e-02 3.19467617377886892518490391523756e-03
            9.68690546064355784139365823648404e-01 1.56547269678229059031160375070613e-02 1.56547269678213099575181388445344e-02 3.19467617377893744676220499911778e-03
            3.34319867363665590431764940149151e-01 6.55493203809416224991934996069176e-01 1.01869288269181845763000637816731e-02 8.11965531899124480508245227383668e-03
            3.34319867363664369186437852476956e-01 1.01869288269104425054267792916107e-02 6.55493203809425217798434459837154e-01 8.11965531898681085187785555490336e-03
            6.55493203809410007742997095192550e-01 3.34319867363671197058039297189680e-01 1.01869288269187951989636076177703e-02 8.11965531899195951115455471835958e-03
            6.55493203809432212203489598323358e-01 1.01869288269144011443989583654002e-02 3.34319867363653377978494063427206e-01 8.11965531898957426637508660860476e-03
            1.01869288269114590533837017005681e-02 3.34319867363656542114114245123346e-01 6.55493203809432101181187135807704e-01 8.11965531898761402884723281658808e-03
            1.01869288269143022651608276873958e-02 6.55493203809432323225792060839012e-01 3.34319867363653377978494063427206e-01 8.11965531898955171496989891011253e-03
            2.92221537796962860245031379236025e-01 5.72337590532031481593833177612396e-01 1.35440871671005713672286674409406e-01 2.68057422831566409537895623316217e-02
            2.92221537796955033172707771882415e-01 1.35440871671014539945332444403903e-01 5.72337590532030482393111014971510e-01 2.68057422831664907136861586423038e-02
            5.72337590532020046296679538500030e-01 2.92221537796963026778485073009506e-01 1.35440871671016926924835388490465e-01 2.68057422831572446375592022604906e-02
            5.72337590532020046296679538500030e-01 1.35440871671038687296118041558657e-01 2.92221537796941266407202419941314e-01 2.68057422831680242092389221397752e-02
            1.35440871671019202882035870061372e-01 2.92221537796959529575957503766404e-01 5.72337590532021378564309088687878e-01 2.68057422831666537776929004621707e-02
            1.35440871671036022760858941182960e-01 5.72337590532032480794555340253282e-01 2.92221537796931496444585718563758e-01 2.68057422831681282926474807482009e-02
            3.19574885423182797339336502773222e-01 6.26001190286244124649783771019429e-01 5.44239242905730780108797262073494e-02 1.84599932108187279877764552793451e-02
            3.19574885423193177924616747986875e-01 5.44239242905576528497313404386659e-02 6.26001190286249120653394584223861e-01 1.84599932108201018887694289105639e-02
            6.26001190286247455318857646489050e-01 3.19574885423173693510534576489590e-01 5.44239242905788511706077770213597e-02 1.84599932108205667946609906948652e-02
            6.26001190286228026415926706249593e-01 5.44239242905720371767941401230928e-02 3.19574885423199950285066961441771e-01 1.84599932108228947935657515699859e-02
            5.44239242905613374023943151769345e-02 3.19574885423196675127144317229977e-01 6.26001190286242015226036983222002e-01 1.84599932108203863834194891069274e-02
            5.44239242905700942865010460991471e-02 6.26001190286225028813760218326934e-01 3.19574885423204890777526543388376e-01 1.84599932108218782456088291610286e-02
            1.90704224192300292317625576288265e-01 7.96427214974067321229256322112633e-01 1.28685608336324142086937172280159e-02 8.47686853432593270807338114991580e-03
            1.90704224192293464446024131575541e-01 1.28685608336373685789411069890775e-02 7.96427214974069097586095722363098e-01 8.47686853432887132964168586113374e-03
            7.96427214974062658292552896455163e-01 1.90704224192302679297128520374827e-01 1.28685608336346624103185831700102e-02 8.47686853432743150915662511124538e-03
            7.96427214974071762121354822738795e-01 1.28685608336418667169143148498733e-02 1.90704224192286359018666530573682e-01 8.47686853433132422863671706636524e-03
            1.28685608336382498184669032070815e-02 1.90704224192287385974964308843482e-01 7.96427214974074315634311460598838e-01 8.47686853432921307016645329213134e-03
            1.28685608336411294594370247068582e-02 7.96427214974073316433589297957951e-01 1.90704224192285498595822446077364e-01 8.47686853433086973108601114290650e-03
            1.80483211648735847321134428966616e-01 7.52351005937754502284064983541612e-01 6.71657824135096781503762031206861e-02 1.82927967700226672131957883493669e-02
            1.80483211648742092325647945472156e-01 6.71657824135280245858581338325166e-02 7.52351005937729966355220767582068e-01 1.82927967700255225680372461738443e-02
            7.52351005937751393659596033103298e-01 1.80483211648729463538742834316508e-01 6.71657824135191428016611325801932e-02 1.82927967700237809056673654595215e-02
            7.52351005937707206783215951872990e-01 6.71657824135427627965100327855907e-02 1.80483211648750030420274015341420e-01 1.82927967700275938278675624815151e-02
            6.71657824135295095091535699793894e-02 1.80483211648748614885917618266831e-01 7.52351005937721861727141003939323e-01 1.82927967700254046068408797509619e-02
            6.71657824135391545716800010268344e-02 7.52351005937705763493283939169487e-01 1.80483211648755026423884828545852e-01 1.82927967700270734108247694393867e-02
            8.07113136795677837964646528234880e-02 9.04625504095602384069252366316505e-01 1.46631822248298737676464043033775e-02 6.66563200416625794425540263432595e-03
            8.07113136795645502719054320550640e-02 1.46631822248288815058181455697195e-02 9.04625504095606602916745941911358e-01 6.66563200416562390282493311133294e-03
            9.04625504095601939980042516253889e-01 8.07113136795689078972770857944852e-02 1.46631822248291521226803979516262e-02 6.66563200416608533926954294202005e-03
            9.04625504095612265054171530209715e-01 1.46631822248259914565071682091002e-02 8.07113136795617469587682535347994e-02 6.66563200416414852050861483689914e-03
            1.46631822248279152648420264881679e-02 8.07113136795616359364657910191454e-02 9.04625504095610488697332129959250e-01 6.66563200416498812667098761153284e-03
            1.46631822248264077901414026428029e-02 9.04625504095612598121078917756677e-01 8.07113136795609698026510159252211e-02 6.66563200416430030881276280751990e-03
            ];
        
    case  18
        
        % ALG. DEG.:   18
        % PTS CARD.:   70
        % NEG. W.  :    3
        % OUT PTS. :    6
        % M.E.INF. : 1.1e-15
        
        xyw_bar=[
            3.33333333333334980164153193982202e-01 3.33333333333333370340767487505218e-01 3.33333333333331593983928087254753e-01 3.08099399376477456335976512491470e-02
            4.93344808630921449754680452315370e-01 4.93344808630920728109714445963618e-01 1.33103827381578221356051017210120e-02 9.07243667940463988097476999428181e-03
            4.93344808630921671799285377346678e-01 1.33103827381572739629866930499702e-02 4.93344808630921061176621833510580e-01 9.07243667940433803908995002984739e-03
            1.33103827381569617127610172246932e-02 4.93344808630922948555763696276699e-01 4.93344808630920061975899670869694e-01 9.07243667940411946393197695215349e-03
            4.69210594241956047145691854893812e-01 4.69210594241956324701448011182947e-01 6.15788115160876836640113651810680e-02 1.87613169395936772720911989154047e-02
            4.69210594241957601457926330112969e-01 6.15788115160862542518671602920222e-02 4.69210594241956213679145548667293e-01 1.87613169395936044137052078895067e-02
            6.15788115160853521956596523523331e-02 4.69210594241957323902170173823833e-01 4.69210594241957323902170173823833e-01 1.87613169395935593108948324925223e-02
            4.36281395887005984945972159039229e-01 4.36281395887005984945972159039229e-01 1.27437208225988030108055681921542e-01 1.94410979854763438234854788788653e-02
            4.36281395887004486144888915077900e-01 1.27437208225988030108055681921542e-01 4.36281395887007539258206634258386e-01 1.94410979854771175101557645348294e-02
            1.27437208225988030108055681921542e-01 4.36281395887005984945972159039229e-01 4.36281395887005984945972159039229e-01 1.94410979854776795605619810203279e-02
            3.94846170673417984353648080286803e-01 3.94846170673417595775589461482014e-01 2.10307658653164419870762458231184e-01 2.77539486108096984662285677813998e-02
            3.94846170673418095375950542802457e-01 2.10307658653166390516631167884043e-01 3.94846170673415541862993904942414e-01 2.77539486108097747940615107609119e-02
            2.10307658653168028095592489989940e-01 3.94846170673415985952203755005030e-01 3.94846170673415985952203755005030e-01 2.77539486108098754080231174157234e-02
            2.49794568803158312242373995104572e-01 2.49794568803156397107656516709540e-01 5.00410862393685262894393872556975e-01 3.22562253514573488155647851272079e-02
            2.49794568803157840397588529413042e-01 5.00410862393686040050511110166553e-01 2.49794568803156091796324744791491e-01 3.22562253514573071822013616838376e-02
            5.00410862393687150273535735323094e-01 2.49794568803157479575105526237166e-01 2.49794568803155370151358738439740e-01 3.22562253514572655488379382404673e-02
            1.61432193743842294919588198354177e-01 1.61432193743842877786676126561360e-01 6.77135612512314799538160059455549e-01 2.50740326169220130447801153650289e-02
            1.61432193743842239408436967096350e-01 6.77135612512314688515857596939895e-01 1.61432193743843099831281051592669e-01 2.50740326169219679419697399680445e-02
            6.77135612512314799538160059455549e-01 1.61432193743842156141710120209609e-01 1.61432193743843044320129820334841e-01 2.50740326169219679419697399680445e-02
            7.65982274853715949802790419198573e-02 7.65982274853715949802790419198573e-02 8.46803545029256810039441916160285e-01 1.52719279718316752747497844211466e-02
            7.65982274853713035467350778162654e-02 8.46803545029257365150954228738556e-01 7.65982274853713729356741168885492e-02 1.52719279718316700705793564907253e-02
            8.46803545029257587195559153769864e-01 7.65982274853712480355838465584384e-02 7.65982274853711647688569996716978e-02 1.52719279718316960914314961428317e-02
            2.42524393534500198876813215065340e-02 2.42524393534499713154239941559354e-02 9.51495121293099987980212972615846e-01 6.79392202296293275243233011906341e-03
            2.42524393534500233571282734601482e-02 9.51495121293100099002515435131500e-01 2.42524393534498949875910511764232e-02 6.79392202296294142604971000309888e-03
            9.51495121293099876957910510100191e-01 2.42524393534500302960221773673766e-02 2.42524393534500927460673125324320e-02 6.79392202296298652886008540008334e-03
            4.31463672169649994914131241330324e-02 4.31463672169649994914131241330324e-02 9.13707265566070070406112790806219e-01 -2.22309872992035219979589655281416e-03
            4.31463672169649994914131241330324e-02 9.13707265566069959383810328290565e-01 4.31463672169650758192460671125446e-02 -2.22309872992043546652274343955469e-03
            9.13707265566069959383810328290565e-01 4.31463672169649994914131241330324e-02 4.31463672169650411247765475764027e-02 -2.22309872992052957527131518133956e-03
            3.58911494940944875153832072101068e-01 6.32657968856635499754759166535223e-01 8.43053620241962509140876136370935e-03 6.33191407640561719494254333540084e-03
            3.58911494940944486575773453296279e-01 8.43053620242006744589513544951842e-03 6.32657968856635499754759166535223e-01 6.33191407640582015758923262183089e-03
            6.32657968856636054866271479113493e-01 3.58911494940944431064622222038452e-01 8.43053620241951406910629884805530e-03 6.33191407640551744834267466899291e-03
            6.32657968856636054866271479113493e-01 8.43053620242038836973819115883089e-03 3.58911494940943542886202521913219e-01 6.33191407640600230355421018657580e-03
            8.43053620242074225332729042747815e-03 3.58911494940944653109227147069760e-01 6.32657968856634611576339466409991e-01 6.33191407640622608288261119469098e-03
            8.43053620242064163936568377266667e-03 6.32657968856636943044691179238725e-01 3.58911494940942432663177896756679e-01 6.33191407640612720364448051668660e-03
            2.94402476751956576617175187493558e-01 5.74410971510856294131031063443515e-01 1.31186551737187073740642517805099e-01 2.72575380491385972636653178824417e-02
            2.94402476751956132527965337430942e-01 1.31186551737187878652335371043591e-01 5.74410971510855961064123675896553e-01 2.72575380491384064440829604336614e-02
            5.74410971510856516175635988474824e-01 2.94402476751956798661780112524866e-01 1.31186551737186685162583899000310e-01 2.72575380491385764469836061607566e-02
            5.74410971510854295729586738161743e-01 1.31186551737188572541725761766429e-01 2.94402476751957131728687500071828e-01 2.72575380491384688941280955987168e-02
            1.31186551737189432964569846262748e-01 2.94402476751956299061419031204423e-01 5.74410971510854295729586738161743e-01 2.72575380491384029746360084800472e-02
            1.31186551737189294186691768118180e-01 5.74410971510854184707284275646089e-01 2.94402476751956521106023956235731e-01 2.72575380491383335856969694077634e-02
            3.25017801641814485513748422818026e-01 6.24779046792512171570876944315387e-01 5.02031515656733429153746328665875e-02 1.76767856494645624054751209541791e-02
            3.25017801641814374491445960302372e-01 5.02031515656742102771126212701347e-02 6.24779046792511394414759706705809e-01 1.76767856494646213860733041656204e-02
            6.24779046792512282593179406831041e-01 3.25017801641814707558353347849334e-01 5.02031515656730098484672453196254e-02 1.76767856494645103637708416499663e-02
            6.24779046792512171570876944315387e-01 5.02031515656749943721237627869414e-02 3.25017801641812820179211485083215e-01 1.76767856494646213860733041656204e-02
            5.02031515656757229559836730459210e-02 3.25017801641813042223816410114523e-01 6.24779046792511283392457244190155e-01 1.76767856494646768972245354234474e-02
            5.02031515656756952004080574170075e-02 6.24779046792511727481667094252771e-01 3.25017801641812598134606560051907e-01 1.76767856494647150611410069132035e-02
            1.84737559666046352591095569550816e-01 7.48933176523037147376271605025977e-01 6.63292638109165277882084410521202e-02 1.83794846380701190624584739907732e-02
            1.84737559666046630146851725839952e-01 6.63292638109162224768766691340716e-02 7.48933176523037147376271605025977e-01 1.83794846380700774290950505474029e-02
            7.48933176523037147376271605025977e-01 1.84737559666046380346671185179730e-01 6.63292638109164722770572097942932e-02 1.83794846380701329402462818052300e-02
            7.48933176523037591465481455088593e-01 6.63292638109162085990888613196148e-02 1.84737559666046213813217491406249e-01 1.83794846380699906929212517070482e-02
            6.63292638109160698212107831750473e-02 1.84737559666046574635700494582125e-01 7.48933176523037369420876530057285e-01 1.83794846380699906929212517070482e-02
            6.63292638109162085990888613196148e-02 7.48933176523037591465481455088593e-01 1.84737559666046213813217491406249e-01 1.83794846380699698762395399853631e-02
            2.18796800013320913436132286733482e-01 7.69207005420443290510945644200547e-01 1.19961945662357960529220690659713e-02 8.10473280819191434221071546062376e-03
            2.18796800013321107725161596135877e-01 1.19961945662357839098577372283216e-02 7.69207005420443068466340719169239e-01 8.10473280819189179080552776213153e-03
            7.69207005420443512555550569231855e-01 2.18796800013320663635951746073260e-01 1.19961945662358238084976846948848e-02 8.10473280819190393386985959978119e-03
            7.69207005420443734600155494263163e-01 1.19961945662358012570924969963926e-02 2.18796800013320469346922436670866e-01 8.10473280819188485191162385490316e-03
            1.19961945662357370723238858545301e-02 2.18796800013321274258615289909358e-01 7.69207005420443068466340719169239e-01 8.10473280819188832135857580851734e-03
            1.19961945662358272779446366484990e-02 7.69207005420443956644760419294471e-01 2.18796800013320247302317511639558e-01 8.10473280819188658663509983171025e-03
            1.01179597136407631308152588189841e-01 8.83962302273466948854263591783820e-01 1.48581005901253782042203965829685e-02 7.63412907072452424223074629594521e-03
            1.01179597136407659063728203818755e-01 1.48581005901254580015002915160949e-02 8.83962302273466837831961129268166e-01 7.63412907072453204848638819157713e-03
            8.83962302273467059876566054299474e-01 1.01179597136407548041425741303101e-01 1.48581005901253920820082043974253e-02 7.63412907072451383388989043510264e-03
            8.83962302273467170898868516815128e-01 1.48581005901254319806481518639885e-02 1.01179597136407395385759855344077e-01 7.63412907072450429291077256266362e-03
            1.48581005901254250417542479567601e-02 1.01179597136407728452667242891039e-01 8.83962302273466837831961129268166e-01 7.63412907072453031376291221477004e-03
            1.48581005901254146334133920959175e-02 8.83962302273467281921170979330782e-01 1.01179597136407339874608624086250e-01 7.63412907072448868039948877139977e-03
            2.08747552825863068315381809725295e-02 1.01434726000536290513309722882695e+00 -3.52220152879492154340823617530987e-02 4.61876607941018673905188285377932e-05
            2.08747552825862998926442770653011e-02 -3.52220152879488823671749742061365e-02 1.01434726000536246104388737876434e+00 4.61876607941034123786146203816116e-05
            1.01434726000536290513309722882695e+00 2.08747552825861784620009586888045e-02 -3.52220152879490835950981875157595e-02 4.61876607941021723223798400859152e-05
            1.01434726000536290513309722882695e+00 -3.52220152879489933894774367217906e-02 2.08747552825860882563802078948356e-02 4.61876607941025856744581001844807e-05
            -3.52220152879488823671749742061365e-02 2.08747552825862860148564692508444e-02 1.01434726000536246104388737876434e+00 4.61876607941034530361960885880279e-05
            -3.52220152879489309394323015567352e-02 1.01434726000536290513309722882695e+00 2.08747552825860882563802078948356e-02 4.61876607941028296199469094229784e-05
            ];
        
    case  19
        
        % ALG. DEG.:   19
        % PTS CARD.:   73
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 8.1e-16
        
        xyw_bar=[
            3.33333333333333481363069950020872e-01 3.33333333333333259318465024989564e-01 3.33333333333333259318465024989564e-01 3.29063313889186165317468635294063e-02
            4.89609987073006791558071881809155e-01 4.89609987073005792357349719168269e-01 2.07800258539873605734271677647484e-02 1.03307318912720744880084922101560e-02
            4.89609987073006069913105875457404e-01 2.07800258539873258789576482286066e-02 4.89609987073006625024618188035674e-01 1.03307318912720571407737324420850e-02
            2.07800258539873015928289845533072e-02 4.89609987073006736046920650551328e-01 4.89609987073005958890803412941750e-01 1.03307318912720050990694531378722e-02
            4.54536892697891881098826161178295e-01 4.54536892697893601944514330170932e-01 9.09262146042145169566595086507732e-02 2.23872472630163958651206002059553e-02
            4.54536892697893157855304480108316e-01 9.09262146042147667568400493109948e-02 4.54536892697891992121128623693949e-01 2.23872472630163715789919365306559e-02
            9.09262146042150304348083977856732e-02 4.54536892697891659054221236146986e-01 4.54536892697893324388758173881797e-01 2.23872472630163854567797443451127e-02
            4.01416680649430701599555959546706e-01 4.01416680649431811822580584703246e-01 1.97166638701137486577863455750048e-01 3.02661258694680787573183522454201e-02
            4.01416680649431589777975659671938e-01 1.97166638701137625355741533894616e-01 4.01416680649430812621858422062360e-01 3.02661258694680718184244483381917e-02
            1.97166638701137958422648921441578e-01 4.01416680649430424043799803257571e-01 4.01416680649431645289126890929765e-01 3.02661258694680891656592081062627e-02
            2.55551654403098071899336218848475e-01 2.55551654403096961676311593691935e-01 4.88896691193805021935503418717417e-01 3.04909678021977742501391617224726e-02
            2.55551654403097461276672675012378e-01 4.88896691193804799890898493686109e-01 2.55551654403097738832428831301513e-01 3.04909678021977742501391617224726e-02
            4.88896691193804688868596031170455e-01 2.55551654403098127410487450106302e-01 2.55551654403097239232067749981070e-01 3.04909678021977707806922097688584e-02
            1.77077942152129880204469714044535e-01 1.77077942152128908759323167032562e-01 6.45844115695741183280631503293989e-01 2.41592127416408869455821673000173e-02
            1.77077942152129214070654938950611e-01 6.45844115695740961236026578262681e-01 1.77077942152129796937742867157795e-01 2.41592127416408904150291192536315e-02
            6.45844115695740961236026578262681e-01 1.77077942152129991226772176560189e-01 1.77077942152129047537201245177130e-01 2.41592127416409147011577829289308e-02
            1.10061053227951316513255619611300e-01 1.10061053227952621025309554170235e-01 7.79877893544095979194707979331724e-01 1.60508035868008255842021014814236e-02
            1.10061053227951996524858202519681e-01 7.79877893544096201239312904363032e-01 1.10061053227951788358041085302830e-01 1.60508035868008672175655249247939e-02
            7.79877893544096534306220291909995e-01 1.10061053227950997324136039878795e-01 1.10061053227952468369643668211211e-01 1.60508035868008880342472366464790e-02
            5.55286242518394645983903501473833e-02 5.55286242518397074596769869003765e-02 8.88942751496320848758614374673925e-01 8.08458026178405890493916530203933e-03
            5.55286242518397421541465064365184e-02 8.88942751496320737736311912158271e-01 5.55286242518394645983903501473833e-02 8.08458026178405023132178541800386e-03
            8.88942751496320737736311912158271e-01 5.55286242518395062317537735907536e-02 5.55286242518397560319343142509751e-02 8.08458026178405196604526139481095e-03
            1.26218637772287159415363078096561e-02 1.26218637772286621651085525286362e-02 9.74756272445542637505866423452972e-01 2.07936202748478784693708654174316e-03
            1.26218637772286083886807972476163e-02 9.74756272445542637505866423452972e-01 1.26218637772287367582180195313413e-02 2.07936202748478437749013458812897e-03
            9.74756272445542637505866423452972e-01 1.26218637772286916554076441343568e-02 1.26218637772286708387259324126717e-02 2.07936202748478307644752760552365e-03
            3.95754787356943027454292405309388e-01 6.00633794794644737535804779327009e-01 3.61141784841223500990281536360271e-03 3.88487690498143158762678339712693e-03
            3.95754787356943027454292405309388e-01 3.61141784841198434236053671497757e-03 6.00633794794644959580409704358317e-01 3.88487690498132707053735579449949e-03
            6.00633794794644848558107241842663e-01 3.95754787356943082965443636567215e-01 3.61141784841206847644912159012165e-03 3.88487690498135916292166136543074e-03
            6.00633794794644959580409704358317e-01 3.61141784841229182209665360403505e-03 3.95754787356942749898536249020253e-01 3.88487690498144763381893618259255e-03
            3.61141784841220595328459275208388e-03 3.95754787356943027454292405309388e-01 6.00633794794644737535804779327009e-01 3.88487690498141120462594066964357e-03
            3.61141784841207107853433555533229e-03 6.00633794794644959580409704358317e-01 3.95754787356942916431989942793734e-01 3.88487690498135048930428148139526e-03
            3.07929983880435520493534795605228e-01 5.57603261588784704372301348485053e-01 1.34466754530779830645315087167546e-01 2.55741606120218831210699761413707e-02
            3.07929983880436741738861883277423e-01 1.34466754530779386556105237104930e-01 5.57603261588783816193881648359820e-01 2.55741606120218865905169280949849e-02
            5.57603261588783483126974260812858e-01 3.07929983880436963783466808308731e-01 1.34466754530779553089558930878411e-01 2.55741606120219004683047359094417e-02
            5.57603261588784260283091498422436e-01 1.34466754530780219223373705972335e-01 3.07929983880435520493534795605228e-01 2.55741606120219212849864476311268e-02
            1.34466754530780191467798090343422e-01 3.07929983880435298448929870573920e-01 5.57603261588784482327696423453745e-01 2.55741606120218935294108320022133e-02
            1.34466754530779913912041934054287e-01 5.57603261588783261082369335781550e-01 3.07929983880436797250013114535250e-01 2.55741606120219004683047359094417e-02
            2.64566948406519730774277832097141e-01 7.20987025817365734248198805289576e-01 1.44460257761145349775233626132831e-02 8.88090357333801323769240099181843e-03
            2.64566948406520674463848763480200e-01 1.44460257761147691651926194822408e-02 7.20987025817364513002871717617381e-01 8.88090357333805313633234845838160e-03
            7.20987025817364624025174180133035e-01 2.64566948406520563441546300964546e-01 1.44460257761148125332795189024182e-02 8.88090357333806527939668029603126e-03
            7.20987025817365179136686492711306e-01 1.44460257761147743693630474126621e-02 2.64566948406520063841185219644103e-01 8.88090357333807395301406018006674e-03
            1.44460257761147275318291960388706e-02 2.64566948406520341396941375933238e-01 7.20987025817364957092081567679998e-01 8.88090357333806180994972834241707e-03
            1.44460257761148559013664183225956e-02 7.20987025817364068913661867554765e-01 2.64566948406521063041907382284990e-01 8.88090357333810517803662776259443e-03
            3.58539352205950012653090652747778e-01 5.94527068955871595967721532360883e-01 4.69335788381783913791878148913383e-02 1.61245467617313759101094206016569e-02
            3.58539352205951067364964046646492e-01 4.69335788381783636236121992624248e-02 5.94527068955870596766999369719997e-01 1.61245467617313793795563725552711e-02
            5.94527068955871040856209219782613e-01 3.58539352205950401231149271552567e-01 4.69335788381785579126415086648194e-02 1.61245467617313932573441803697278e-02
            5.94527068955870818811604294751305e-01 4.69335788381786758738378750877018e-02 3.58539352205950512253451734068221e-01 1.61245467617314175434728440450272e-02
            4.69335788381785093403841813142208e-02 3.58539352205951067364964046646492e-01 5.94527068955870374722394444688689e-01 1.61245467617313828490033245088853e-02
            4.69335788381786966905195868093870e-02 5.94527068955869819610882132110419e-01 3.58539352205951455943022665451281e-01 1.61245467617314244823667479522555e-02
            1.57807405968594077760869254234422e-01 8.39331473680839379447604642336955e-01 2.86112035056651503595048779970966e-03 2.49194181749063867203908273495472e-03
            1.57807405968594993694864569988567e-01 2.86112035056767339755157131264696e-03 8.39331473680837381046160317055183e-01 2.49194181749094051392390269938915e-03
            8.39331473680837936157672629633453e-01 1.57807405968594993694864569988567e-01 2.86112035056707014746280037797987e-03 2.49194181749079739923713461280386e-03
            8.39331473680839046380697254789993e-01 2.86112035056589183654174313176100e-03 1.57807405968595049206015801246394e-01 2.49194181749045175558454623399030e-03
            2.86112035056642656505321298254785e-03 1.57807405968594993694864569988567e-01 8.39331473680838602291487404727377e-01 2.49194181749060267652695621620751e-03
            2.86112035056694091056384010585134e-03 8.39331473680837381046160317055183e-01 1.57807405968595659828679345082492e-01 2.49194181749076227108674608246019e-03
            7.50505969759109881067615788197145e-02 7.01087978926172805671512833214365e-01 2.23861424097916206221725587965921e-01 1.82428401189506060597889103291891e-02
            7.50505969759108076955200772317767e-02 2.23861424097916178466149972337007e-01 7.01087978926173027716117758245673e-01 1.82428401189505678958724388394330e-02
            7.01087978926173249760722683276981e-01 7.50505969759110713734884257064550e-02 2.23861424097915678865788891016564e-01 1.82428401189505921820011025147323e-02
            7.01087978926173693849932533339597e-01 2.23861424097915318043305887840688e-01 7.50505969759109881067615788197145e-02 1.82428401189505783042132947002756e-02
            2.23861424097915789888091353532218e-01 7.50505969759107660621566537884064e-02 7.01087978926173471805327608308289e-01 1.82428401189505505486376790713621e-02
            2.23861424097915373554457119098515e-01 7.01087978926173804872234995855251e-01 7.50505969759108770844591163040604e-02 1.82428401189505436097437751641337e-02
            1.42421601113383283765401188247779e-01 8.22931324069856917624576908565359e-01 3.46470748167597708544462875579484e-02 1.02585637361984911147816390553089e-02
            1.42421601113382922942918185071903e-01 3.46470748167610753665002221168834e-02 8.22931324069855918423854745924473e-01 1.02585637361984460119712636583245e-02
            8.22931324069856362513064595987089e-01 1.42421601113383228254249956989952e-01 3.46470748167604092326854470229591e-02 1.02585637361984824411642591712734e-02
            8.22931324069857250691484296112321e-01 3.46470748167589451260717225977714e-02 1.42421601113383811121337885197136e-01 1.02585637361985813204023898492778e-02
            3.46470748167595626876291703410971e-02 1.42421601113383672343459807052568e-01 8.22931324069856806602274446049705e-01 1.02585637361985223398042066378366e-02
            3.46470748167603259659586001362186e-02 8.22931324069856140468459670955781e-01 1.42421601113383533565581728908001e-01 1.02585637361985466259328703131359e-02
            6.54946280829374338772197461366886e-02 9.24344252620784345708671025931835e-01 1.01611192962782759252604591893032e-02 3.79992885530188112888638762854043e-03
            6.54946280829379334775808274571318e-02 1.01611192962782828641543630965316e-02 9.24344252620783790597158713353565e-01 3.79992885530192406329241805451602e-03
            9.24344252620783790597158713353565e-01 6.54946280829379057220052118282183e-02 1.01611192962783036808360748182167e-02 3.79992885530192102752633509510360e-03
            9.24344252620784123664066100900527e-01 1.01611192962782394960674636763542e-02 6.54946280829376420440368633535400e-02 3.79992885530189370563158846039187e-03
            1.01611192962781649029579966736492e-02 6.54946280829377808219149414981075e-02 9.24344252620784012641763638384873e-01 3.79992885530189544035506443719896e-03
            1.01611192962782915377717429805671e-02 9.24344252620783568552553788322257e-01 6.54946280829381555221857524884399e-02 3.79992885530194531365499877040293e-03
            ];
        
    case  20
        
        % ALG. DEG.:   20
        % PTS CARD.:   79
        % NEG. W.  :    3
        % OUT PTS. :    9
        % M.E.INF. : 9.9e-16
        
        xyw_bar=[
            3.33333333333333425851918718763045e-01 3.33333333333333314829616256247391e-01 3.33333333333333203807313793731737e-01 3.30570555416238864854250323332963e-02
            5.00950464352196256356819503707811e-01 5.00950464352196478401424428739119e-01 -1.90092870439273475824393244693056e-03 8.67019185663472024237774515853516e-04
            5.00950464352200031115103229240049e-01 -1.90092870439298304054143162744595e-03 5.00950464352192925687745628238190e-01 8.67019185663432884539347789143449e-04
            -1.90092870439260855711105513421444e-03 5.00950464352200031115103229240049e-01 5.00950464352192481598535778175574e-01 8.67019185663481890477544133943866e-04
            4.88212957934727465048041494810604e-01 4.88212957934727409536890263552777e-01 2.35740841305451809262194728944451e-02 1.16600527164471088475972138098768e-02
            4.88212957934727798114948882357567e-01 2.35740841305451219456212896830039e-02 4.88212957934727020958831644747988e-01 1.16600527164471296642789255315620e-02
            2.35740841305452641929463197811856e-02 4.88212957934727742603797651099740e-01 4.88212957934726965447680413490161e-01 1.16600527164471071128737378330698e-02
            4.55136681950281951625214560408494e-01 4.55136681950281951625214560408494e-01 8.97266360994360412384196479251841e-02 2.28769363564207128813876579442876e-02
            4.55136681950282062647517022924148e-01 8.97266360994361106273586869974679e-02 4.55136681950281896114063329150667e-01 2.28769363564207232897285138051302e-02
            8.97266360994361522607221104408382e-02 4.55136681950282007136365791666321e-01 4.55136681950281785091760866635013e-01 2.28769363564207163508346098979018e-02
            4.01996259318289195228146581939654e-01 4.01996259318289139716995350681827e-01 1.96007481363421665054858067378518e-01 3.04489826739379673692376115923253e-02
            4.01996259318289195228146581939654e-01 1.96007481363421748321584914265259e-01 4.01996259318289084205844119424000e-01 3.04489826739379777775784674531678e-02
            1.96007481363421831588311761151999e-01 4.01996259318289195228146581939654e-01 4.01996259318288973183541656908346e-01 3.04489826739379708386845635459395e-02
            2.55892909759421249837885170563823e-01 2.55892909759421249837885170563823e-01 4.88214180481157500324229658872355e-01 3.06248917253546457106772038514464e-02
            2.55892909759421194326733939305996e-01 4.88214180481157666857683352645836e-01 2.55892909759421194326733939305996e-01 3.06248917253546491801241558050606e-02
            4.88214180481157777879985815161490e-01 2.55892909759421194326733939305996e-01 2.55892909759421083304431476790342e-01 3.06248917253546387717832999442180e-02
            1.76488255995106035900477081668214e-01 1.76488255995106063656052697297127e-01 6.47023488009787928199045836663572e-01 2.43680576768004197862893533965689e-02
            1.76488255995105980389325850410387e-01 6.47023488009788039221348299179226e-01 1.76488255995105980389325850410387e-01 2.43680576768003955001606897212696e-02
            6.47023488009788150243650761694880e-01 1.76488255995106008144901466039300e-01 1.76488255995105841611447772265819e-01 2.43680576768004093779484975357263e-02
            1.04170855336758461096025030201417e-01 1.04170855336758461096025030201417e-01 7.91658289326483188830252402112819e-01 1.59974320320239206394408171263422e-02
            1.04170855336758405584873798943590e-01 7.91658289326483188830252402112819e-01 1.04170855336758405584873798943590e-01 1.59974320320239102310999612654996e-02
            7.91658289326483299852554864628473e-01 1.04170855336758363951510375500220e-01 1.04170855336758336195934759871307e-01 1.59974320320239032922060573582712e-02
            5.30689638409298294896565550970990e-02 5.30689638409297947951870355609572e-02 8.93862072318140299387323466362432e-01 7.69830181560240967009178802982206e-03
            5.30689638409295935672638222513342e-02 8.93862072318140299387323466362432e-01 5.30689638409300723509431918500923e-02 7.69830181560230732140670539820348e-03
            8.93862072318140521431928391393740e-01 5.30689638409294686671735519212234e-02 5.30689638409300099008980566850369e-02 7.69830181560238451660138636611919e-03
            4.16187151960289991592389924335293e-02 4.16187151960289991592389924335293e-02 9.16762569607942001681522015132941e-01 -6.32060497487685313787453278422390e-04
            4.16187151960239615222647557857272e-02 9.16762569607942001681522015132941e-01 4.16187151960339951628498056379613e-02 -6.32060497487594023964529998949047e-04
            9.16762569607942001681522015132941e-01 4.16187151960243986725807019411150e-02 4.16187151960335996458972829259437e-02 -6.32060497487676098068987151634701e-04
            1.15819214068221820745918648754014e-02 1.15819214068222063607205285507007e-02 9.76836157186355524828513807733543e-01 1.75113430119276810258133192377272e-03
            1.15819214068221699315275330377517e-02 9.76836157186355635850816270249197e-01 1.15819214068222375857430961332284e-02 1.75113430119276246473003499914967e-03
            9.76836157186355635850816270249197e-01 1.15819214068222028912735765970865e-02 1.15819214068221612579101531537162e-02 1.75113430119276376577264198175499e-03
            3.44855770229001101778720794754918e-01 6.06402646106159570749127851740923e-01 4.87415836648392719610001222463325e-02 1.64658391895757504064246035113683e-02
            3.44855770229001268312174488528399e-01 4.87415836648393274721513535041595e-02 6.06402646106159348704522926709615e-01 1.64658391895757677536593632794393e-02
            6.06402646106159681771430314256577e-01 3.44855770229001101778720794754918e-01 4.87415836648392164498488909885054e-02 1.64658391895757399980837476505258e-02
            6.06402646106159792793732776772231e-01 4.87415836648393829833025847619865e-02 3.44855770229000824222964638465783e-01 1.64658391895757434675306996041400e-02
            4.87415836648393344110452574113879e-02 3.44855770229001268312174488528399e-01 6.06402646106159348704522926709615e-01 1.64658391895757573453185074185967e-02
            4.87415836648393899221964886692149e-02 6.06402646106159792793732776772231e-01 3.44855770229000824222964638465783e-01 1.64658391895757399980837476505258e-02
            3.77843269594853925497091040597297e-01 6.15842614456540871081813293130836e-01 6.31411594860520342109566627186723e-03 4.83903354048480314336666552321731e-03
            3.77843269594855035720115665753838e-01 6.31411594860513576688010317639055e-03 6.15842614456539871881091130489949e-01 4.83903354048480314336666552321731e-03
            6.15842614456540871081813293130836e-01 3.77843269594853981008242271855124e-01 6.31411594860514790994443501404021e-03 4.83903354048479967391971356960312e-03
            6.15842614456541537215628068224760e-01 6.31411594860525893224689752969425e-03 3.77843269594853203852125034245546e-01 4.83903354048478492877016776674282e-03
            6.31411594860514010368879311840828e-03 3.77843269594855035720115665753838e-01 6.15842614456539871881091130489949e-01 4.83903354048480834753709345363859e-03
            6.31411594860526326905558747171199e-03 6.15842614456541537215628068224760e-01 3.77843269594853148340973802987719e-01 4.83903354048479446974928563918183e-03
            3.06635479062356774981878970720572e-01 5.59048000390295452000088971544756e-01 1.34316520547347773018032057734672e-01 2.58049065346499933992774344915233e-02
            3.06635479062356830493030201978399e-01 1.34316520547347800773607673363585e-01 5.59048000390295340977786509029102e-01 2.58049065346500038076182903523659e-02
            5.59048000390295563022391434060410e-01 3.06635479062356719470727739462745e-01 1.34316520547347717506880826476845e-01 2.58049065346500003381713383987517e-02
            5.59048000390295563022391434060410e-01 1.34316520547347773018032057734672e-01 3.06635479062356663959576508204918e-01 2.58049065346499933992774344915233e-02
            1.34316520547347800773607673363585e-01 3.06635479062356830493030201978399e-01 5.59048000390295340977786509029102e-01 2.58049065346500142159591462132084e-02
            1.34316520547347800773607673363585e-01 5.59048000390295563022391434060410e-01 3.06635479062356663959576508204918e-01 2.58049065346499933992774344915233e-02
            2.49419362774742109367664966157463e-01 7.36606743262865681920459337561624e-01 1.39738939623922364674513119098265e-02 8.47109105444064126866354769163081e-03
            2.49419362774742525701299200591166e-01 1.39738939623921532007244650230859e-02 7.36606743262865348853551950014662e-01 8.47109105444065341172787952928047e-03
            7.36606743262865792942761800077278e-01 2.49419362774742109367664966157463e-01 1.39738939623920976895732337652589e-02 8.47109105444064126866354769163081e-03
            7.36606743262866014987366725108586e-01 1.39738939623921601396183689303143e-02 2.49419362774741831811908809868328e-01 8.47109105444061698253488401633149e-03
            1.39738939623921306493192773245937e-02 2.49419362774742525701299200591166e-01 7.36606743262865348853551950014662e-01 8.47109105444065861589830745970175e-03
            1.39738939623921497312775130694718e-02 7.36606743262865903965064262592932e-01 2.49419362774741970589786888012895e-01 8.47109105444062045198183596994568e-03
            2.12775724802801630142212729879247e-01 7.11675142287434225885078831197461e-01 7.55491329097641717282840545522049e-02 1.83549141062797314949417426532818e-02
            2.12775724802801741164515192394902e-01 7.55491329097641856060718623666617e-02 7.11675142287434114862776368681807e-01 1.83549141062797453727295504677386e-02
            7.11675142287434336907381293713115e-01 2.12775724802801602386637114250334e-01 7.55491329097640607059815920365509e-02 1.83549141062797314949417426532818e-02
            7.11675142287434447929683756228769e-01 7.55491329097641439727084389232914e-02 2.12775724802801408097607804847939e-01 1.83549141062797280254947906996676e-02
            7.55491329097641856060718623666617e-02 2.12775724802801713408939576765988e-01 7.11675142287434114862776368681807e-01 1.83549141062797488421765024213528e-02
            7.55491329097641439727084389232914e-02 7.11675142287434336907381293713115e-01 2.12775724802801491364334651734680e-01 1.83549141062797314949417426532818e-02
            1.46965436053239223435085136770795e-01 8.61402717154987396241949682007544e-01 -8.36815320822659192145920314942487e-03 7.04404677908219197662864630160584e-04
            1.46965436053239223435085136770795e-01 -8.36815320822662141175829475514547e-03 8.61402717154987396241949682007544e-01 7.04404677908215294535043682344622e-04
            8.61402717154987396241949682007544e-01 1.46965436053239167923933905512968e-01 -8.36815320822656416588358752051136e-03 7.04404677908213885072219451188857e-04
            8.61402717154987951353461994585814e-01 -8.36815320822675498546594496929174e-03 1.46965436053238807101450902337092e-01 7.04404677908204777773970572951612e-04
            -8.36815320822661620758786682472419e-03 1.46965436053239223435085136770795e-01 8.61402717154987507264252144523198e-01 7.04404677908216378737216167849056e-04
            -8.36815320822669427014428578104344e-03 8.61402717154987729308857069554506e-01 1.46965436053238973634904596110573e-01 7.04404677908210198784833000473782e-04
            1.37726978828923041220377854187973e-01 8.35586957912362815470430632558418e-01 2.66860632587141433091915132536087e-02 1.01126849274618922647217544863452e-02
            1.37726978828923263264982779219281e-01 2.66860632587140808591463780885533e-02 8.35586957912362704448128170042764e-01 1.01126849274619078772330382776090e-02
            8.35586957912362926492733095074072e-01 1.37726978828922985709226622930146e-01 2.66860632587140877980402819957817e-02 1.01126849274619026730626103471877e-02
            8.35586957912362926492733095074072e-01 2.66860632587140218785481948771121e-02 1.37726978828923041220377854187973e-01 1.01126849274618766522104706950813e-02
            2.66860632587141086147219937174668e-02 1.37726978828923235509407163590367e-01 8.35586957912362593425825707527110e-01 1.01126849274619148161269421848374e-02
            2.66860632587140565730177144132540e-02 8.35586957912362815470430632558418e-01 1.37726978828923152242680316703627e-01 1.01126849274618714480400427646600e-02
            5.96961091490066306231021542316739e-02 9.29756171556852639703549812111305e-01 1.05477192941407782456053610076196e-02 3.57390938595026578378233139687836e-03
            5.96961091490065751119509229738469e-02 1.05477192941408511039913520335176e-02 9.29756171556852528681247349595651e-01 3.57390938595032823382746656193376e-03
            9.29756171556852750725852274626959e-01 5.96961091490065196007996917160199e-02 1.05477192941407296733480336570210e-02 3.57390938595027142163362832150142e-03
            9.29756171556852639703549812111305e-01 1.05477192941409447790590547811007e-02 5.96961091490064155173911331075942e-02 3.57390938595036162725437911547033e-03
            1.05477192941408910026312995000808e-02 5.96961091490064918452240760871064e-02 9.29756171556852639703549812111305e-01 3.57390938595033083591268052714440e-03
            1.05477192941409551873999106419433e-02 9.29756171556852639703549812111305e-01 5.96961091490063600062399018497672e-02 3.57390938595036553038220006328629e-03
            ];
        
        
    otherwise
        xyw=[];
end




pointset_stats=[
    1    1 1.92e-16    0    0
    2    3 3.68e-16    0    0
    3    4 2.22e-16    1    0
    4    6 2.01e-16    0    0
    5    7 3.15e-16    0    0
    6   12 3.12e-16    0    0
    7   13 3.75e-16    1    0
    8   16 6.11e-16    0    0
    9   19 3.19e-16    0    0
    10   25 5.07e-16    0    0
    11   27 8.88e-16    0    3
    12   33 3.89e-16    0    0
    13   37 4.79e-16    0    0
    14   42 8.47e-16    0    0
    15   48 8.47e-16    0    9
    16   52 9.02e-16    0    6
    17   61 5.64e-16    0    0
    18   70 1.11e-15    3    6
    19   73 8.10e-16    0    0
    20   79 9.91e-16    3    9
    ];


