
function [xyw,pointset_stats]=set_dunavant_C_standard(deg)

%--------------------------------------------------------------------
% CUBATURE RULE ON THE SIMPLEX [0 0; 0 1; 1 0] WITH ADE EQUAL TO deg.
% THE CUBATURE RULE IS DESCRIBED IN STANDARD FORM.
%--------------------------------------------------------------------
% INPUT:
% deg: algebraic degree of precision (ADE) of the rule.
%--------------------------------------------------------------------
% OUTPUT:
% xyw: MATRIX OF NODES xyw(:,1:2) AND WEIGHTS xyw(:,3).
%      REMARK: The sum of the weights makes 1/2.
%
% pointset_stats:
%    statistics of the pointset stored in a matrix. Each rows contains
%    degree, cardinality of the full pointset, moment error w.r.t. orthn.
%    Dubiner basis,number of points with negative weights, number of
%    points outside the domain.
%--------------------------------------------------------------------
% SOURCE:
%  D.A. Dunavant,
%  "High Degree Efficient Symmetrical Gaussian Quadrature Rules for the
%  Triangle"
%  International Journal for Numerical Methods in Engineering,
%  21(6), June 1985, pp. 1129-1148.
%--------------------------------------------------------------------
% | DEG | CARD | W- | PT.O|   M.E.   |Q.|
%--------------------------------------------------------------------
% |   1 |   1 |   0 |   0 | 1.92e-16 |PI|
% |   2 |   3 |   0 |   0 | 3.68e-16 |PI|
% |   3 |   4 |   1 |   0 | 2.22e-16 |NI|
% |   4 |   6 |   0 |   0 | 2.01e-16 |PI|
% |   5 |   7 |   0 |   0 | 3.15e-16 |PI|
% |   6 |  12 |   0 |   0 | 3.12e-16 |PI|
% |   7 |  13 |   1 |   0 | 3.75e-16 |NI|
% |   8 |  16 |   0 |   0 | 6.11e-16 |PI|
% |   9 |  19 |   0 |   0 | 3.19e-16 |PI|
% |  10 |  25 |   0 |   0 | 5.07e-16 |PI|
% |  11 |  27 |   0 |   3 | 8.88e-16 |P0|
% |  12 |  33 |   0 |   0 | 3.89e-16 |PI|
% |  13 |  37 |   0 |   0 | 4.79e-16 |PI|
% |  14 |  42 |   0 |   0 | 8.47e-16 |PI|
% |  15 |  48 |   0 |   9 | 8.47e-16 |P0|
% |  16 |  52 |   0 |   6 | 9.02e-16 |P0|
% |  17 |  61 |   0 |   0 | 5.64e-16 |PI|
% |  18 |  70 |   3 |   6 | 1.11e-15 |N0|
% |  19 |  73 |   0 |   0 | 8.10e-16 |PI|
% |  20 |  79 |   3 |   9 | 9.91e-16 |N0|
%--------------------------------------------------------------------
% LEGEND:
%--------------------------------------------------------------------
% DEG  : algebraic degree of precision
% CARD : cardinality of the pointset
% W-   : negative weights
% PT.O : points outside the simplex [0 0; 1 0; 0 1].
% M.E. : maximum moment error in inf norm.
% Q.: quality of the rule.
%        PI: pos. weights, points in simplex
%        NI: some neg. weights, points in simplex
%        P0: pos. weights, some points out of the simplex
%        N0: some neg. weights, some points out of the simplex
%--------------------------------------------------------------------

switch deg
    
    case   1
        
        % ALG. DEG.:    1
        % PTS CARD.:    1
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.9e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 5.00000000000000000000000000000000e-01
            ];
        
    case   2
        
        % ALG. DEG.:    2
        % PTS CARD.:    3
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.7e-16
        
        xyw=[
            1.66666666666666657414808128123695e-01 1.66666666666666657414808128123695e-01 1.66666666666666657414808128123695e-01
            1.66666666666666657414808128123695e-01 6.66666666666666740681534975010436e-01 1.66666666666666657414808128123695e-01
            6.66666666666666740681534975010436e-01 1.66666666666666657414808128123695e-01 1.66666666666666657414808128123695e-01
            ];
        
    case   3
        
        % ALG. DEG.:    3
        % PTS CARD.:    4
        % NEG. W.  :    1
        % OUT PTS. :    0
        % M.E.INF. : 2.2e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 -2.81250000000000000000000000000000e-01
            2.00000000000000011102230246251565e-01 2.00000000000000011102230246251565e-01 2.60416666666666685170383743752609e-01
            2.00000000000000011102230246251565e-01 5.99999999999999977795539507496869e-01 2.60416666666666685170383743752609e-01
            5.99999999999999977795539507496869e-01 2.00000000000000011102230246251565e-01 2.60416666666666685170383743752609e-01
            ];
        
    case   4
        
        % ALG. DEG.:    4
        % PTS CARD.:    6
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 2.0e-16
        
        xyw=[
            4.45948490915965001235576892213430e-01 4.45948490915964779190971967182122e-01 1.11690794839005735905601568447310e-01
            4.45948490915965001235576892213430e-01 1.08103018168070233451238948418904e-01 1.11690794839005735905601568447310e-01
            1.08103018168070219573451140604448e-01 4.45948490915965001235576892213430e-01 1.11690794839005749783389376261766e-01
            9.15762135097707846709269574603240e-02 9.15762135097707846709269574603240e-02 5.49758718276609978370395026558981e-02
            9.15762135097708124265025730892376e-02 8.16847572980458402902570469450438e-01 5.49758718276609423258882713980711e-02
            8.16847572980458624947175394481746e-01 9.15762135097706875264123027591268e-02 5.49758718276608937536309440474724e-02
            ];
        
    case   5
        
        % ALG. DEG.:    5
        % PTS CARD.:    7
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.2e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333370340767487505218e-01 1.12500000000000002775557561562891e-01
            4.70142064105115164984738385101082e-01 4.70142064105115109473587153843255e-01 6.61970763942530959766941123234574e-02
            4.70142064105115053962435922585428e-01 5.97158717897698365639769235713175e-02 6.61970763942530959766941123234574e-02
            5.97158717897698157472952118496323e-02 4.70142064105115109473587153843255e-01 6.61970763942530820989063045090006e-02
            1.01286507323456301254971378966729e-01 1.01286507323456342888334802410100e-01 6.29695902724135836425745083033689e-02
            1.01286507323456342888334802410100e-01 7.97426985353087314223330395179801e-01 6.29695902724135975203623161178257e-02
            7.97426985353087203201027932664147e-01 1.01286507323456356766122610224556e-01 6.29695902724135836425745083033689e-02
            ];
        
    case   6
        
        % ALG. DEG.:    6
        % PTS CARD.:   12
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.1e-16
        
        xyw=[
            2.49286745170910012392440080475353e-01 2.49286745170912538149821102706483e-01 5.83931378631896216835350799101434e-02
            2.49286745170910012392440080475353e-01 5.01426509658180030726271070307121e-01 5.83931378631894273945057705077488e-02
            5.01426509658180030726271070307121e-01 2.49286745170910012392440080475353e-01 5.83931378631900449560632182510744e-02
            6.30890144915022543781191188827506e-02 6.30890144915020878446654251092696e-02 2.54224531851033712370746542319466e-02
            6.30890144915021294780288485526398e-02 8.73821971016995768799517918523634e-01 2.54224531851033434814990386030331e-02
            8.73821971016995213688005605945364e-01 6.30890144915025180560874673574290e-02 2.54224531851035516483161558198844e-02
            5.31450498448169314502642635034135e-02 3.10352451033784004774673803694895e-01 4.14255378091867923484592495242396e-02
            5.31450498448167510390227619154757e-02 6.36502499121398890302714335120982e-01 4.14255378091867298984141143591842e-02
            3.10352451033784004774673803694895e-01 5.31450498448177224841693089274486e-02 4.14255378091872780710325230302260e-02
            3.10352451033783893752371341179241e-01 6.36502499121399001325016797636636e-01 4.14255378091867854095653456170112e-02
            6.36502499121398557235806947574019e-01 5.31450498448165498110995486058528e-02 4.14255378091865703038543244929315e-02
            6.36502499121397891101992172480095e-01 3.10352451033785337042303353882744e-01 4.14255378091866327538994596579869e-02
            ];
        
    case   7
        
        % ALG. DEG.:    7
        % PTS CARD.:   13
        % NEG. W.  :    1
        % OUT PTS. :    0
        % M.E.INF. : 3.7e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 -7.47850222338408837652323768452334e-02
            2.60345966079039703444664155540522e-01 2.60345966079039925489269080571830e-01 8.78076287166039137765949362801621e-02
            2.60345966079039981000420311829657e-01 4.79308067841920260043764301371993e-01 8.78076287166039415321705519090756e-02
            4.79308067841920426577217995145475e-01 2.60345966079039647933512924282695e-01 8.78076287166038860210193206512486e-02
            6.51301029022158389780727816287254e-02 6.51301029022158112224971659998118e-02 2.66736178044192501213061063936038e-02
            6.51301029022158112224971659998118e-02 8.69739794195568349799430052371463e-01 2.66736178044192466518591544399897e-02
            8.69739794195568349799430052371463e-01 6.51301029022158389780727816287254e-02 2.66736178044192466518591544399897e-02
            4.86903154253163772402501763281180e-02 3.12865496004873788749023333366495e-01 3.85568804451285607193256055325037e-02
            4.86903154253164327514014075859450e-02 6.38444188569809667477272796531906e-01 3.85568804451286092915829328831023e-02
            3.12865496004873899771325795882149e-01 4.86903154253164396902953114931734e-02 3.85568804451286023526890289758740e-02
            3.12865496004873844260174564624322e-01 6.38444188569809778499575259047560e-01 3.85568804451285676582195094397321e-02
            6.38444188569809778499575259047560e-01 4.86903154253163703013562724208896e-02 3.85568804451285468415377977180469e-02
            6.38444188569809667477272796531906e-01 3.12865496004873899771325795882149e-01 3.85568804451286162304768367903307e-02
            ];
        
    case   8
        
        % ALG. DEG.:    8
        % PTS CARD.:   16
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.1e-16
        
        xyw=[
            3.33333333333333370340767487505218e-01 3.33333333333333314829616256247391e-01 7.21578038388935583125061157261371e-02
            4.59292588292722903098308506741887e-01 4.59292588292723402698669588062330e-01 4.75458171336423096597911808203207e-02
            4.59292588292723180654064663031022e-01 8.14148234145535970585072504945856e-02 4.75458171336422888431094690986356e-02
            8.14148234145537913475365598969802e-02 4.59292588292723014120610969257541e-01 4.75458171336423304764728925420059e-02
            1.70569307751760351754555244951916e-01 1.70569307751760018687647857404954e-01 5.16086852673591292561816601391911e-02
            1.70569307751760157465525935549522e-01 6.58861384496479685068948128900956e-01 5.16086852673591292561816601391911e-02
            6.58861384496479463024343203869648e-01 1.70569307751760323998979629323003e-01 5.16086852673591431339694679536478e-02
            5.05472283170309635846884077636787e-02 5.05472283170309844013701194853638e-02 1.62292488115990361785545559314414e-02
            5.05472283170309219513249843203084e-02 8.98905543365938086708410992287099e-01 1.62292488115990361785545559314414e-02
            8.98905543365938086708410992287099e-01 5.05472283170309913402640233925922e-02 1.62292488115990396480015078850556e-02
            8.39477740995770924103958066098130e-03 2.63112829634638223375020515959477e-01 1.36151570872175258636316286242618e-02
            8.39477740995758260622583435406341e-03 7.28492392955404022281129527982557e-01 1.36151570872174998427794889721554e-02
            2.63112829634638001330415590928169e-01 8.39477740995748025754075172244484e-03 1.36151570872174530052456375983638e-02
            2.63112829634637945819264359670342e-01 7.28492392955404577392641840560827e-01 1.36151570872174634135864934592064e-02
            7.28492392955404466370339378045173e-01 8.39477740995771617993348456820968e-03 1.36151570872175223941846766706476e-02
            7.28492392955404022281129527982557e-01 2.63112829634638223375020515959477e-01 1.36151570872175223941846766706476e-02
            ];
        
    case   9
        
        % ALG. DEG.:    9
        % PTS CARD.:   19
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.2e-16
        
        xyw=[
            3.33333333333332537673499018637813e-01 3.33333333333333925452279800083488e-01 4.85678981413993279825369597801910e-02
            4.89682519198738286370087280374719e-01 4.89682519198736898591306498929043e-01 1.56673501135696051356216429439883e-02
            4.89682519198738008814331124085584e-01 2.06349616025247942219245089745527e-02 1.56673501135695739105990753614606e-02
            2.06349616025248254469470765570804e-02 4.89682519198737342680516348991659e-01 1.56673501135695392161295558253187e-02
            4.37089591492935969352373604124296e-01 4.37089591492937024064246998023009e-01 3.89137705023871252607570170312101e-02
            4.37089591492936135885827297897777e-01 1.25820817014127089850106244739436e-01 3.89137705023872154663777678251790e-02
            1.25820817014126506983018316532252e-01 4.37089591492936968553095766765182e-01 3.89137705023870072995606506083277e-02
            1.88203535619032219505797343117592e-01 1.88203535619032996661914580727171e-01 3.98238694636051035469215264583909e-02
            1.88203535619032163994646111859765e-01 6.23592928761935172410346694960026e-01 3.98238694636050688524520069222490e-02
            6.23592928761933729120414682256524e-01 1.88203535619033246462095121387392e-01 3.98238694636051659969666616234463e-02
            4.47295133944527190106299485705676e-02 4.47295133944527051328421407561109e-02 1.27888378293490156262235402095939e-02
            4.47295133944527745217811798283947e-02 9.10540973211094506467588871601038e-01 1.27888378293490347081817759544720e-02
            9.10540973211094728512193796632346e-01 4.47295133944526704383726212199690e-02 1.27888378293489826664774966502591e-02
            3.68384120547361887387793899506505e-02 2.21962989160765650220596967301390e-01 2.16417696886446186965713422978297e-02
            3.68384120547361054720525430639100e-02 7.41198598784498008384957756788936e-01 2.16417696886446013493365825297587e-02
            2.21962989160765872265201892332698e-01 3.68384120547363830278086993530451e-02 2.16417696886447331883207567670979e-02
            2.21962989160765983287504354848352e-01 7.41198598784497786340352831757627e-01 2.16417696886446672688286696484283e-02
            7.41198598784498008384957756788936e-01 3.68384120547365218056867774976126e-02 2.16417696886447678827902763032398e-02
            7.41198598784498230429562681820244e-01 2.21962989160765400420416426641168e-01 2.16417696886447227799799009062554e-02
            ];
        
    case  10
        
        % ALG. DEG.:   10
        % PTS CARD.:   25
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.1e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 4.54089951913767692071033366119082e-02
            4.85577633383657003562916543160100e-01 4.85577633383657725207882549511851e-01 1.83629788782333734642460143504650e-02
            4.85577633383657003562916543160100e-01 2.88447332326852573514130995135929e-02 1.83629788782333457086703987215515e-02
            2.88447332326852330652844358382936e-02 4.85577633383657003562916543160100e-01 1.83629788782333318308825909070947e-02
            1.09481575485037757600181862471800e-01 1.09481575485036522477066966985149e-01 2.26605297177639511407853945001989e-02
            1.09481575485036189410159579438186e-01 7.81036849029925872578417056502076e-01 2.26605297177639650185732023146556e-02
            7.81036849029926094623021981533384e-01 1.09481575485037480044425706182665e-01 2.26605297177639788963610101291124e-02
            1.41707219414880319918736972795159e-01 3.07939838764120288328030028424109e-01 3.63789584227100573632363023079961e-02
            1.41707219414879487251468503927754e-01 5.50352941820998697863842608057894e-01 3.63789584227100018520850710501691e-02
            3.07939838764121620595659578611958e-01 1.41707219414879626029346582072321e-01 3.63789584227100157298728788646258e-02
            3.07939838764120177305727565908455e-01 5.50352941820999475019959845667472e-01 3.63789584227101059354936296585947e-02
            5.50352941820999363997657383151818e-01 1.41707219414880347674312588424073e-01 3.63789584227100781799180140296812e-02
            5.50352941820998808886145070573548e-01 3.07939838764121565084508347354131e-01 3.63789584227100573632363023079961e-02
            2.50035347626866161219183481989603e-02 2.46672560639902560541969478435931e-01 1.41636212655288223905003519575985e-02
            2.50035347626861442771328825074306e-02 7.28323904597410698436021903034998e-01 1.41636212655286281014710425552039e-02
            2.46672560639902394008515784662450e-01 2.50035347626861338687920266465881e-02 1.41636212655286419792588503696606e-02
            2.46672560639902199719486475260055e-01 7.28323904597411142525231753097614e-01 1.41636212655288570849698714937404e-02
            7.28323904597410365369114515488036e-01 2.50035347626865606107671169411333e-02 1.41636212655288414724585877024765e-02
            7.28323904597411031502929290581960e-01 2.46672560639902782586574403467239e-01 1.41636212655286680001109900217671e-02
            9.54081540029997135499151283966057e-03 6.68032510122005263042410661000758e-02 4.71083348186650350403192888393278e-03
            9.54081540029878827358089665722218e-03 9.23655933587500643788814613799332e-01 4.71083348186629186776785971346726e-03
            6.68032510121999989483043691507191e-02 9.54081540029910052380657248249918e-03 4.71083348186632829696085522641624e-03
            6.68032510121999989483043691507191e-02 9.23655933587499866632697376189753e-01 4.71083348186653646377797244326757e-03
            9.23655933587499533565789988642791e-01 9.54081540029977012706829953003762e-03 4.71083348186648008526500319703700e-03
            9.23655933587500976855722001346294e-01 6.68032510121999989483043691507191e-02 4.71083348186633697057823511045171e-03
            ];
        
    case  11
        
        % ALG. DEG.:   11
        % PTS CARD.:   27
        % NEG. W.  :    0
        % OUT PTS. :    3
        % M.E.INF. : 8.9e-16
        
        xyw=[
            5.34611048270758226763632592337672e-01 5.34611048270758337785935054853326e-01 4.63503164480338620927202253341193e-04
            5.34611048270758559830539979884634e-01 -6.92220965415166339385066862632812e-02 4.63503164480337861985681513488089e-04
            -6.92220965415166061829310706343676e-02 5.34611048270758337785935054853326e-01 4.63503164480338295666550507689863e-04
            3.98969302965854977216508814308327e-01 3.98969302965855421305718664370943e-01 3.85747674574065460562088958340610e-02
            3.98969302965854977216508814308327e-01 2.02061394068289573722196905691817e-01 3.85747674574065460562088958340610e-02
            2.02061394068289518211045674433990e-01 3.98969302965854977216508814308327e-01 3.85747674574065529951027997412893e-02
            2.03309900431282303268076816493704e-01 2.03309900431282192245774353978049e-01 2.96614886903870077217160172722288e-02
            2.03309900431282303268076816493704e-01 5.93380199137435004885787748207804e-01 2.96614886903870215995038250866855e-02
            5.93380199137434560796577898145188e-01 2.03309900431282719601711050927406e-01 2.96614886903870493550794407155990e-02
            1.19350912282581225754007903105958e-01 1.19350912282581253509583518734871e-01 1.80922702517089910301262278835566e-02
            1.19350912282581517187551867209550e-01 7.61298175434837243180652421870036e-01 1.80922702517090500107244110949978e-02
            7.61298175434837132158349959354382e-01 1.19350912282581433920825020322809e-01 1.80922702517090916440878345383680e-02
            3.23649481112758610756863220103696e-02 3.23649481112758610756863220103696e-02 6.82986550133892443559258111918098e-03
            3.23649481112758957701558415465115e-02 9.35270103777448236215263932535890e-01 6.82986550133892790503953307279517e-03
            9.35270103777448236215263932535890e-01 3.23649481112758957701558415465115e-02 6.82986550133893484393343698002354e-03
            5.01781383104946410167457315765205e-02 3.56620648261292472191996694164118e-01 2.61685559811020151954608081723563e-02
            5.01781383104946548945335393909772e-02 5.93201213428212636991077033599140e-01 2.61685559811020256038016640331989e-02
            3.56620648261292583214299156679772e-01 5.01781383104946340778518276692921e-02 2.61685559811020186649077601259705e-02
            3.56620648261292527703147925421945e-01 5.93201213428212859035681958630448e-01 2.61685559811020429510364238012698e-02
            5.93201213428212636991077033599140e-01 5.01781383104947034667908667415759e-02 2.61685559811020498899303277084982e-02
            5.93201213428212636991077033599140e-01 3.56620648261292638725450387937599e-01 2.61685559811020429510364238012698e-02
            2.10220165361662721104085704837416e-02 1.71488980304041471747211744514061e-01 1.03538298195703353660190870755287e-02
            2.10220165361663657854762732313247e-02 8.07489003159791995933858288481133e-01 1.03538298195703683257651306348635e-02
            1.71488980304041555013938591400802e-01 2.10220165361663310910067536951829e-02 1.03538298195703457743599429363712e-02
            1.71488980304041527258362975771888e-01 8.07489003159792217978463213512441e-01 1.03538298195703197535078032842648e-02
            8.07489003159792106956160750996787e-01 2.10220165361663102743250419734977e-02 1.03538298195703405701895150059499e-02
            8.07489003159792217978463213512441e-01 1.71488980304041555013938591400802e-01 1.03538298195703423049129909827570e-02
            ];
        
    case  12
        
        % ALG. DEG.:   12
        % PTS CARD.:   33
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.9e-16
        
        xyw=[
            4.88217389773805021668806602974655e-01 4.88217389773804744113050446685520e-01 1.28655332202276557335141049520644e-02
            4.88217389773805188202260296748136e-01 2.35652204523902238098020944789823e-02 1.28655332202276609376845328824857e-02
            2.35652204523902342181429503398249e-02 4.88217389773805021668806602974655e-01 1.28655332202276678765784367897140e-02
            4.39724392294460586505522314837435e-01 4.39724392294459975882858771001338e-01 2.18462722690192130692299343763807e-02
            4.39724392294460142416312464774819e-01 1.20551215411079423733831106346770e-01 2.18462722690192130692299343763807e-02
            1.20551215411079465367194529790140e-01 4.39724392294460808550127239868743e-01 2.18462722690191783747604148402388e-02
            2.71210385012115751113270789574017e-01 2.71210385012116195202480639636633e-01 3.14291121089425307122411368254689e-02
            2.71210385012116028669026945863152e-01 4.57579229975768331240004727078485e-01 3.14291121089425376511350407326972e-02
            4.57579229975767998173097339531523e-01 2.71210385012115862135573252089671e-01 3.14291121089425237733472329182405e-02
            1.27576145541585861398203860517242e-01 1.27576145541586000176081938661810e-01 1.73980564653544826636544229359060e-02
            1.27576145541585916909355091775069e-01 7.44847708916828055158987353934208e-01 1.73980564653544618469727112142209e-02
            7.44847708916828388225894741481170e-01 1.27576145541585778131477013630501e-01 1.73980564653544375608440475389216e-02
            2.13173504532104124420399671180348e-02 2.13173504532104228503808229788774e-02 3.08313052577951964700053721912809e-03
            2.13173504532104124420399671180348e-02 9.57365299093579258382646912650671e-01 3.08313052577950403448925342786424e-03
            9.57365299093579258382646912650671e-01 2.13173504532103569308887358602078e-02 3.08313052577950533553186041046956e-03
            1.15343494534697874054351984796085e-01 2.75713269685514628104527901086840e-01 2.01857788831904944037987092997355e-02
            1.15343494534698040587805678569566e-01 6.08943235779787928052542156365234e-01 2.01857788831904146065188143666091e-02
            2.75713269685514017481864357250743e-01 1.15343494534698207121259372343047e-01 2.01857788831904735871169975780504e-02
            2.75713269685514295037620513539878e-01 6.08943235779787928052542156365234e-01 2.01857788831904701176700456244362e-02
            6.08943235779787928052542156365234e-01 1.15343494534697971198866639497282e-01 2.01857788831904874649048053925071e-02
            6.08943235779787928052542156365234e-01 2.75713269685513906459561894735089e-01 2.01857788831904666482230936708220e-02
            2.28383322222570142590036823548871e-02 2.81325580989939649789732811768772e-01 1.11783866011517067640124167837712e-02
            2.28383322222570489534732018910290e-02 6.95836086787803620445913566072704e-01 1.11783866011517345195880324126847e-02
            2.81325580989939871834337736800080e-01 2.28383322222570697701549136127142e-02 1.11783866011517466626523642503344e-02
            2.81325580989939705300884043026599e-01 6.95836086787803287379006178525742e-01 1.11783866011516998251185128765428e-02
            6.95836086787803620445913566072704e-01 2.28383322222570107895567304012729e-02 1.11783866011517119681828447141925e-02
            6.95836086787803509423611103557050e-01 2.81325580989939427745127886737464e-01 1.11783866011517362543115083894918e-02
            2.57340505483302020572899948547274e-02 1.16251915907597244692617266537127e-01 8.65811555432944093824154663252557e-03
            2.57340505483302159350778026691842e-02 8.58014033544072796111379375361139e-01 8.65811555432943920351807065571847e-03
            1.16251915907597300203768497794954e-01 2.57340505483302367517595143908693e-02 8.65811555432945828547630640059651e-03
            1.16251915907597286325980689980497e-01 8.58014033544072463044471987814177e-01 8.65811555432945828547630640059651e-03
            8.58014033544072796111379375361139e-01 2.57340505483302089961838987619558e-02 8.65811555432943573407111870210429e-03
            8.58014033544072685089076912845485e-01 1.16251915907597078159163572763646e-01 8.65811555432942706045373881806881e-03
            ];
        
    case  13
        
        % ALG. DEG.:   13
        % PTS CARD.:   37
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.8e-16
        
        xyw=[
            3.33333333333333592385372412536526e-01 3.33333333333334036474582262599142e-01 2.62604617004010970626737275779305e-02
            4.95048184939703816986877882300178e-01 4.95048184939705426810263588777161e-01 5.64007260466474220211852141915188e-03
            4.95048184939704982721053738714545e-01 9.90363012059061048608654687086528e-03 5.64007260466474133475678343074833e-03
            9.90363012059072150838900938651932e-03 4.95048184939704982721053738714545e-01 5.64007260466476822297066107125829e-03
            4.68716635109574675244203945112531e-01 4.68716635109573842576935476245126e-01 1.57117591812269870310547759117981e-02
            4.68716635109573787065784244987299e-01 6.25667297808519817792216599627864e-02 1.57117591812271709117432294533501e-02
            6.25667297808519123902826208905026e-02 4.68716635109573731554633013729472e-01 1.57117591812270807061224786593812e-02
            4.14521336801277617833960675852722e-01 4.14521336801276951700145900758798e-01 2.35362512520971671470260133673946e-02
            4.14521336801276618633238513211836e-01 1.70957326397446957022552282978722e-01 2.35362512520972677609876200222061e-02
            1.70957326397446429666615586029366e-01 4.14521336801277062722448363274452e-01 2.35362512520972261276241965788358e-02
            2.29399572042831012552710490126628e-01 2.29399572042831734197676496478380e-01 2.36817932681773618552867333164613e-02
            2.29399572042831512153071571447072e-01 5.41200855914337974894579019746743e-01 2.36817932681773826719684450381465e-02
            5.41200855914337974894579019746743e-01 2.29399572042831456641920340189245e-01 2.36817932681773514469458774556188e-02
            1.14424495196330244795035468996502e-01 1.14424495196330078261581775223021e-01 1.55837645228969218919035100157089e-02
            1.14424495196330369695125739326613e-01 7.71151009607339288365324136975687e-01 1.55837645228969669947138854126933e-02
            7.71151009607339732454533987038303e-01 1.14424495196329759072462195490516e-01 1.55837645228969253613504619693231e-02
            2.48113913634589455803336477401899e-02 2.48113913634589906831440231371744e-02 3.98788573253718199584927361911468e-03
            2.48113913634589559886745036010325e-02 9.50377217273082108839332704519620e-01 3.98788573253717418959363172348276e-03
            9.50377217273081997817030242003966e-01 2.48113913634590253776135426733163e-02 3.98788573253718546529622557272887e-03
            9.48538283795782588558864745209576e-02 2.68794997058761031549778408589191e-01 1.84242013643661217692226728104288e-02
            9.48538283795784809004913995522656e-02 6.36351174561660126727247188682668e-01 1.84242013643661287081165767176572e-02
            2.68794997058761031549778408589191e-01 9.48538283795787862118231714703143e-02 1.84242013643661321775635286712713e-02
            2.68794997058761697683593183683115e-01 6.36351174561660348771852113713976e-01 1.84242013643660731969653454598301e-02
            6.36351174561659793660339801135706e-01 9.48538283795786613117329011402035e-02 1.84242013643660974830940091351295e-02
            6.36351174561661347972574276354862e-01 2.68794997058760809505173483557883e-01 1.84242013643660905442001052279011e-02
            1.81007732788068115281632231017284e-02 2.91730066734287862839636318312841e-01 8.70073165191097806780984313945737e-03
            1.81007732788068878559961660812405e-02 6.90169159986905533799017575802281e-01 8.70073165191102143589674255963473e-03
            2.91730066734287696306182624539360e-01 1.81007732788069572449352051535243e-02 8.70073165191105786508973807258371e-03
            2.91730066734287363239275236992398e-01 6.90169159986905866865924963349244e-01 8.70073165191095378168117946415805e-03
            6.90169159986905533799017575802281e-01 1.81007732788069919394047246896662e-02 8.70073165191105959981321404939081e-03
            6.90169159986905200732110188255319e-01 2.91730066734288251417694937117631e-01 8.70073165191091908721165992801616e-03
            2.22330766740901838995903005979926e-02 1.26357385491668749999405463313451e-01 7.76089341952247626504890476439869e-03
            2.22330766740901596134616369226933e-02 8.51409537834241225695564025954809e-01 7.76089341952248233658107068322352e-03
            1.26357385491668638977103000797797e-01 2.22330766740900936939695498040237e-02 7.76089341952247019351673884557385e-03
            1.26357385491668527954800538282143e-01 8.51409537834241336717866488470463e-01 7.76089341952247192824021482238095e-03
            8.51409537834241447740168950986117e-01 2.22330766740900208355835587781257e-02 7.76089341952244157057938522825680e-03
            8.51409537834240892628656638407847e-01 1.26357385491668916532859157086932e-01 7.76089341952249794909235447448737e-03
            ];
        
    case  14
        
        % ALG. DEG.:   14
        % PTS CARD.:   42
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 8.5e-16
        
        xyw=[
            4.88963910362178899582374924648320e-01 4.88963910362178399982013843327877e-01 1.09417906847144412452177419936561e-02
            4.88963910362178344470862612070050e-01 2.20721792756427559467624632816296e-02 1.09417906847144481841116459008845e-02
            2.20721792756427316606337996063303e-02 4.88963910362178955093526155906147e-01 1.09417906847144395104942660168490e-02
            4.17644719340453995926054631127045e-01 4.17644719340453995926054631127045e-01 1.63941767720626602189426535005623e-02
            4.17644719340453995926054631127045e-01 1.64710561319092452237100587808527e-01 1.63941767720626463411548456861055e-02
            1.64710561319092008147890737745911e-01 4.17644719340453995926054631127045e-01 1.63941767720626949134121730367042e-02
            2.73477528308838646609046918456443e-01 2.73477528308838702120198149714270e-01 2.58870522536457751960892181841700e-02
            2.73477528308838646609046918456443e-01 4.53044943382322706781906163087115e-01 2.58870522536457994822178818594693e-02
            4.53044943382322595759603700571461e-01 2.73477528308838813142500612229924e-01 2.58870522536457578488544584160991e-02
            1.77205532412543192988252371833369e-01 1.77205532412543720344189068782725e-01 2.10812943684965010959775355559032e-02
            1.77205532412543526055159759380331e-01 6.45588935174913225445436637528474e-01 2.10812943684964976265305836022890e-02
            6.45588935174913114423134175012819e-01 1.77205532412543304010554834349023e-01 2.10812943684965080348714394631315e-02
            6.17998830908723581489816467637866e-02 6.17998830908729618327512866926554e-02 7.21684983488834162768110758179319e-03
            6.17998830908726218269499952384649e-02 8.76400233818255158801946436142316e-01 7.21684983488832081099939586010805e-03
            8.76400233818254492668131661048392e-01 6.17998830908724830490719170938974e-02 7.21684983488834769921327350061802e-03
            1.93909612487010336867587767528676e-02 1.93909612487010336867587767528676e-02 2.46170180120004137430678170517240e-03
            1.93909612487010267478648728456392e-02 9.61218077502597911809800734772580e-01 2.46170180120002879756158087332096e-03
            9.61218077502597911809800734772580e-01 1.93909612487010683812282962890094e-02 2.46170180120004657847720963559368e-03
            5.71247574036476968295517053775256e-02 1.72266687821355873877493536383554e-01 1.23328766062818159787717320341471e-02
            5.71247574036479952019895733883459e-02 7.70608554774996790115437761414796e-01 1.23328766062818385301769197326394e-02
            1.72266687821355485299434917578765e-01 5.71247574036482380632762101413391e-02 1.23328766062818454690708236398677e-02
            1.72266687821355596321737380094419e-01 7.70608554774996568070832836383488e-01 1.23328766062818350607299677790252e-02
            7.70608554774996123981622986320872e-01 5.71247574036478772407932069654635e-02 1.23328766062818246523891119181826e-02
            7.70608554774996457048530373867834e-01 1.72266687821355513055010533207678e-01 1.23328766062818298565595398486039e-02
            9.29162493569716391172974567780329e-02 3.36861459796345241723969365921221e-01 1.92857553935303106806831152653103e-02
            9.29162493569718472841145739948843e-02 5.70222290846683521614579603919992e-01 1.92857553935303384362587308942238e-02
            3.36861459796344908657061978374259e-01 9.29162493569720276953560755828221e-02 1.92857553935303766001752023839799e-02
            3.36861459796345075190515672147740e-01 5.70222290846683188547672216373030e-01 1.92857553935303314973648269869955e-02
            5.70222290846682966503067291341722e-01 9.29162493569718056507511505515140e-02 1.92857553935303245584709230797671e-02
            5.70222290846683188547672216373030e-01 3.36861459796344853145910747116432e-01 1.92857553935303349668117789406097e-02
            1.46469500556543723301938797476396e-02 2.98372882136258066321232718109968e-01 7.21815405676689073044549616042787e-03
            1.46469500556544174330042551446240e-02 6.86980167808088126868426570581505e-01 7.21815405676692369019153971976266e-03
            2.98372882136257511209720405531698e-01 1.46469500556545093733484819154000e-02 7.21815405676694103742629948783360e-03
            2.98372882136258010810081486852141e-01 6.86980167808087571756914258003235e-01 7.21815405676691761865937380093783e-03
            6.86980167808087571756914258003235e-01 1.46469500556544347802390149126950e-02 7.21815405676691241448894587051655e-03
            6.86980167808088015846124108065851e-01 2.98372882136257511209720405531698e-01 7.21815405676691328185068385892009e-03
            1.26833093287180413179238147591832e-03 1.18974497696956907355492205624614e-01 2.50511441925030343622071704601240e-03
            1.26833093287203485001468639126188e-03 8.79757171370171286994832371419761e-01 2.50511441925033466124328462854010e-03
            1.18974497696956782455401935294503e-01 1.26833093287230134690868332825175e-03 2.50511441925037239147888712409440e-03
            1.18974497696956935111067821253528e-01 8.79757171370171175972529908904107e-01 2.50511441925030690566766899962658e-03
            8.79757171370170953927924983872799e-01 1.26833093287197239996955122620648e-03 2.50511441925032512026416675610108e-03
            8.79757171370171175972529908904107e-01 1.18974497696956671433099472778849e-01 2.50511441925036111577629327484829e-03
            ];
        
    case  15
        
        % ALG. DEG.:   15
        % PTS CARD.:   48
        % NEG. W.  :    0
        % OUT PTS. :    9
        % M.E.INF. : 8.5e-16
        
        xyw=[
            5.06972916858242972359960276662605e-01 5.06972916858242972359960276662605e-01 9.58437821424309240210537286230874e-04
            5.06972916858242972359960276662605e-01 -1.39458337164859204337918896499104e-02 9.58437821424310758093578765937082e-04
            -1.39458337164858666573641343688905e-02 5.06972916858242972359960276662605e-01 9.58437821424311517035099505790186e-04
            4.31406354283023041329414581923629e-01 4.31406354283022208662146113056224e-01 2.21245136355723605314782531650053e-02
            4.31406354283022763773658425634494e-01 1.37187291433954777764014920649061e-01 2.21245136355723674703721570722337e-02
            1.37187291433954750008439305020147e-01 4.31406354283023207862868275697110e-01 2.21245136355723605314782531650053e-02
            2.77693644847144427423302204260835e-01 2.77693644847144538445604666776489e-01 2.55932743594260610209278894444651e-02
            2.77693644847144482934453435518662e-01 4.44612710305711256175698053993983e-01 2.55932743594260540820339855372367e-02
            4.44612710305711311686849285251810e-01 2.77693644847144316400999741745181e-01 2.55932743594260575514809374908509e-02
            1.26464891041253996162296857619367e-01 1.26464891041254440251506707681983e-01 1.18438679353439300229178243739625e-02
            1.26464891041254023917872473248281e-01 7.47070217917492507275767366081709e-01 1.18438679353438970631717808146277e-02
            7.47070217917492951364977216144325e-01 1.26464891041253385539633313783270e-01 1.18438679353438935937248288610135e-02
            7.08083859746862082085172573897580e-02 7.08083859746859722861245245439932e-02 6.64488784501030601420401566770124e-03
            7.08083859746855559524902901102905e-02 8.58383228050628277472355875943322e-01 6.64488784501024182943540452583875e-03
            8.58383228050628610539263263490284e-01 7.08083859746857918748830229560554e-02 6.64488784501020973705109895490750e-03
            1.89651702410733209724913450600070e-02 1.89651702410733452586200087353063e-02 2.37445830409592842369370657706895e-03
            1.89651702410733730141956243642198e-02 9.62069659517853281727184366900474e-01 2.37445830409592061743806468143703e-03
            9.62069659517853392749486829416128e-01 1.89651702410732689307870657557942e-02 2.37445830409591714799111272782284e-03
            1.33734161966621162287438551175001e-01 2.61311371140088066766793417627923e-01 1.92750362997962526590001175463840e-02
            1.33734161966621106776287319917174e-01 6.04954466893292019946670734498184e-01 1.92750362997962561284470694999982e-02
            2.61311371140087567166432336307480e-01 1.33734161966621190043014166803914e-01 1.92750362997962630673409734072266e-02
            2.61311371140087789211037261338788e-01 6.04954466893291242790553496888606e-01 1.92750362997962526590001175463840e-02
            6.04954466893291686879763346951222e-01 1.33734161966620995753984857401520e-01 1.92750362997962595978940214536124e-02
            6.04954466893292019946670734498184e-01 2.61311371140087012054920023729210e-01 1.92750362997962561284470694999982e-02
            3.63666773969168327651502181652177e-02 5.75586555512814679325117594999028e-01 1.36079071603121340428010910272860e-02
            3.63666773969168813374075455158163e-02 3.88046767090269084654607922857394e-01 1.36079071603121305733541390736718e-02
            5.75586555512814346258210207452066e-01 3.63666773969168188873624103507609e-02 1.36079071603121323080776150504789e-02
            5.75586555512814568302815132483374e-01 3.88046767090268585054246841536951e-01 1.36079071603121236344602351664435e-02
            3.88046767090268751587700535310432e-01 3.63666773969168188873624103507609e-02 1.36079071603121375122480429809002e-02
            3.88046767090269029143456691599567e-01 5.75586555512814124213605282420758e-01 1.36079071603121305733541390736718e-02
            -1.01748831265706645571311739217890e-02 2.85712220049916032493797501956578e-01 1.09103868339851715028865353218634e-03
            -1.01748831265708397642022475793055e-02 7.24462663076654744820359610457672e-01 1.09103868339850153777736974092250e-03
            2.85712220049915976982646270698751e-01 -1.01748831265706662918546498985961e-02 1.09103868339851758396952252638812e-03
            2.85712220049915976982646270698751e-01 7.24462663076654522775754685426364e-01 1.09103868339852625758690241042359e-03
            7.24462663076654633798057147942018e-01 -1.01748831265706385362790342696826e-02 1.09103868339851476504387406407659e-03
            7.24462663076654744820359610457672e-01 2.85712220049915976982646270698751e-01 1.09103868339850695878823216844467e-03
            3.68438698758782462450156458544370e-02 2.15599664072284286087111127017124e-01 1.07526599238656934975910672847021e-02
            3.68438698758781282838192794315546e-02 7.47556466051838008901597731892252e-01 1.07526599238656865586971633774738e-02
            2.15599664072284008531354970727989e-01 3.68438698758782740005912614833505e-02 1.07526599238657056406553991223518e-02
            2.15599664072284230575959895759297e-01 7.47556466051837564812387881829636e-01 1.07526599238656674767389276325957e-02
            7.47556466051837786856992806860944e-01 3.68438698758781282838192794315546e-02 1.07526599238656657420154516557886e-02
            7.47556466051837897879295269376598e-01 2.15599664072283925264628123841248e-01 1.07526599238656674767389276325957e-02
            1.24598093311987720421818437444017e-02 1.03575616576385864942011494349572e-01 3.83697131552434818682617567731086e-03
            1.24598093311986072434516259477277e-02 8.83964574092415800166122608061414e-01 3.83697131552430611978188323973882e-03
            1.03575616576385753919709031833918e-01 1.24598093311986766323906650200115e-02 3.83697131552432737014446395562572e-03
            1.03575616576385823308648070906202e-01 8.83964574092415356076912757998798e-01 3.83697131552436293197572148017116e-03
            8.83964574092415467099215220514452e-01 1.24598093311987546949470839763308e-02 3.83697131552434385001748573529312e-03
            8.83964574092415578121517683030106e-01 1.03575616576385656775194377132721e-01 3.83697131552432390069751200201154e-03
            ];
        
    case  16
        
        % ALG. DEG.:   16
        % PTS CARD.:   52
        % NEG. W.  :    0
        % OUT PTS. :    6
        % M.E.INF. : 9.0e-16
        
        xyw=[
            3.33333333333333259318465024989564e-01 3.33333333333333370340767487505218e-01 2.34378487138208052575905071535090e-02
            4.97380541948438004151711311351391e-01 4.97380541948438781307828548960970e-01 3.20293928929247509115874059659745e-03
            4.97380541948438004151711311351391e-01 5.23891610312323102033316146730613e-03 3.20293928929248376477612048063293e-03
            5.23891610312315555986195647619752e-03 4.97380541948438004151711311351391e-01 3.20293928929247552483960959079923e-03
            4.13469438549352197309616485654260e-01 4.13469438549352474865372641943395e-01 2.08551483696934290834601455344455e-02
            4.13469438549352252820767716912087e-01 1.73061122901295300069435256773431e-01 2.08551483696934152056723377199887e-02
            1.73061122901295105780405947371037e-01 4.13469438549352363843070179427741e-01 2.08551483696934186751192896736029e-02
            4.70458599066991278725424763251795e-01 4.70458599066991500770029688283103e-01 1.34457421250322017164391397159307e-02
            4.70458599066991112191971069478313e-01 5.90828018660173107101662992590718e-02 1.34457421250322051858860916695448e-02
            5.90828018660172274434394523723313e-02 4.70458599066991112191971069478313e-01 1.34457421250321861039278559246668e-02
            2.40553749969520863638550167706853e-01 2.40553749969520919149701398964680e-01 2.10662613808248050384364091769385e-02
            2.40553749969520752616247705191199e-01 5.18892500060958328234050895844121e-01 2.10662613808248189162242169913952e-02
            5.18892500060958106189445970812812e-01 2.40553749969520974660852630222507e-01 2.10662613808248154467772650377810e-02
            1.47965794222572888383382405663724e-01 1.47965794222572777361079943148070e-01 1.50001334213864731537135810413019e-02
            1.47965794222572666338777480632416e-01 7.04068411554854223233235188672552e-01 1.50001334213864852967779128789516e-02
            7.04068411554854112210932726156898e-01 1.47965794222572832872231174405897e-01 1.50001334213864887662248648325658e-02
            7.54651876574742169756504495126137e-02 7.54651876574742169756504495126137e-02 7.10004946251207788576742530040065e-03
            7.54651876574740365644089479246759e-02 8.49069624685051649315425947861513e-01 7.10004946251209002883175713805031e-03
            8.49069624685051316248518560314551e-01 7.54651876574743002423772963993542e-02 7.10004946251209610036392305687514e-03
            1.65964026230248709592451916705613e-02 1.65964026230249403481842307428451e-02 1.79123117563667719372155762158627e-03
            1.65964026230249230009494709747742e-02 9.66807194753950160936994961957680e-01 1.79123117563668044632807507809957e-03
            9.66807194753950160936994961957680e-01 1.65964026230249715732067983253728e-02 1.79123117563668174737068206070489e-03
            1.03575692245251996248533998823405e-01 2.96555596579887281727394565677969e-01 1.63865737303136860258945262103225e-02
            1.03575692245251912981807151936664e-01 5.99868711174860402834951855766121e-01 1.63865737303136929647884301175509e-02
            2.96555596579887337238545796935796e-01 1.03575692245252051759685230081232e-01 1.63865737303136929647884301175509e-02
            2.96555596579887337238545796935796e-01 5.99868711174860624879556780797429e-01 1.63865737303136999036823340247793e-02
            5.99868711174860291812649393250467e-01 1.03575692245252024004109614452318e-01 1.63865737303136999036823340247793e-02
            5.99868711174860513857254318281775e-01 2.96555596579887503771999490709277e-01 1.63865737303137068425762379320076e-02
            2.00834116554158628731308056103444e-02 3.37723063403078782584998407401145e-01 7.64915312422057794977670042158024e-03
            2.00834116554158455258960458422735e-02 6.42193524941504723102525531430729e-01 7.64915312422058228658539036359798e-03
            3.37723063403078782584998407401145e-01 2.00834116554158732814716614711870e-02 7.64915312422058575603234231721217e-03
            3.37723063403078893607300869916799e-01 6.42193524941505167191735381493345e-01 7.64915312422059269492624622444055e-03
            6.42193524941504834124827993946383e-01 2.00834116554158455258960458422735e-02 7.64915312422058835811755628242281e-03
            6.42193524941504834124827993946383e-01 3.37723063403079337696510719979415e-01 7.64915312422059009284103225922991e-03
            -4.34100261413866445742160138365762e-03 2.04748281642811641445334203126549e-01 1.19312209641935147198810174273831e-03
            -4.34100261413873211163716447913430e-03 7.99592720971326609280538377788616e-01 1.19312209641934605097723931521614e-03
            2.04748281642811696956485434384376e-01 -4.34100261413874251997802033997687e-03 1.19312209641934128048768037899663e-03
            2.04748281642811946756665975044598e-01 7.99592720971326609280538377788616e-01 1.19312209641936838554199251660748e-03
            7.99592720971326831325143302819924e-01 -4.34100261413875466304235217762653e-03 1.19312209641934128048768037899663e-03
            7.99592720971326609280538377788616e-01 2.04748281642811974512241590673511e-01 1.19312209641936144664808860937910e-03
            4.19417864680099103380506164739927e-02 1.89358492130622363402991936709441e-01 9.54239637794943029724947791692102e-03
            4.19417864680098131935359617727954e-02 7.68699721401367441764307386620203e-01 9.54239637794944070559033377776359e-03
            1.89358492130622363402991936709441e-01 4.19417864680098478880054813089373e-02 9.54239637794943897086685780095650e-03
            1.89358492130622280136265089822700e-01 7.68699721401367663808912311651511e-01 9.54239637794944417503728573137778e-03
            7.68699721401367552786609849135857e-01 4.19417864680098270713237695872522e-02 9.54239637794945458337814159222035e-03
            7.68699721401367330742004924104549e-01 1.89358492130622613203172477369662e-01 9.54239637794945458337814159222035e-03
            1.43173202306813591305756361293788e-02 8.52836156826571323286145798192592e-02 3.42502727327098936346838620181643e-03
            1.43173202306813331097234964772724e-02 9.00399064086661371497655181883601e-01 3.42502727327098762874491022500933e-03
            8.52836156826570351840999251180619e-02 1.43173202306814285195146752016626e-02 3.42502727327099847076663508005367e-03
            8.52836156826571739619780032626295e-02 9.00399064086661482519957644399256e-01 3.42502727327099673604315910324658e-03
            9.00399064086661260475352719367947e-01 1.43173202306814458667494349697336e-02 3.42502727327100671070314596988737e-03
            9.00399064086661371497655181883601e-01 8.52836156826572572287048501493700e-02 3.42502727327100107285184904526432e-03
            ];
        
    case  17
        
        % ALG. DEG.:   17
        % PTS CARD.:   61
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.6e-16
        
        xyw=[
            3.33333333333340642301578782280558e-01 3.33333333333348802440809777181130e-01 1.67185996454037727465546936400642e-02
            4.97170540556762186046313445331180e-01 4.97170540556774009921525703248335e-01 2.54670772025673104385634637480962e-03
            4.97170540556774009921525703248335e-01 5.65891888646769675164094337560527e-03 2.54670772025732388560426144863413e-03
            5.65891888645994600715027900150744e-03 4.97170540556774009921525703248335e-01 2.54670772025512512359846084564197e-03
            4.82176322624617181844541846658103e-01 4.82176322624609909883730551882763e-01 7.33543226381833547017352614716401e-03
            4.82176322624613629130863046157174e-01 3.56473547507641097542574470935506e-02 7.33543226381509500672040147151165e-03
            3.56473547507499821662690919765737e-02 4.82176322624618181045264009298990e-01 7.33543226381540985903129126199929e-03
            4.50239969020782893061038976156851e-01 4.50239969020782004882619276031619e-01 1.21754391768311258370527383476656e-02
            4.50239969020783448172551288735121e-01 9.95200619584295093078551985854574e-02 1.21754391768348225327800449235838e-02
            9.95200619584220708135902100366366e-02 4.50239969020793828757831533948774e-01 1.21754391768360004100202331756009e-02
            4.00266239377405741439730491038063e-01 4.00266239377414734246229954806040e-01 1.55537754344820151886086279091614e-02
            4.00266239377403465482530009467155e-01 1.99467521245205864355298785994819e-01 1.55537754344874986495161905963869e-02
            1.99467521245202922264283529329987e-01 4.00266239377414512201625029774732e-01 1.55537754344883122348264237189142e-02
            2.52141267970952986932076100856648e-01 2.52141267970952986932076100856648e-01 1.56285556093143859768268555399118e-02
            2.52141267970967974942908540469944e-01 4.95717464058113954639139819846605e-01 1.56285556093114369469176949678513e-02
            4.95717464058097578849526598787634e-01 2.52141267970972249301553347322624e-01 1.56285556093113363329560883130398e-02
            1.62047004658467025395296445822169e-01 1.62047004658467747040262452173920e-01 1.24078271698326546290669369909665e-02
            1.62047004658484705696963601440075e-01 6.75905990683077995129224291304126e-01 1.24078271698322452343266064644922e-02
            6.75905990683063118140694314206485e-01 1.62047004658487370232222701815772e-01 1.24078271698322209481979427891929e-02
            7.58758822607437238838556936570967e-02 7.58758822607484978428615818302205e-02 7.02803653527853035981598495141043e-03
            7.58758822607369376456176723877434e-02 8.48248235478509693408000202907715e-01 7.02803653527844188891871013424861e-03
            8.48248235478513801233191315986915e-01 7.58758822607351196554148486939084e-02 7.02803653527810882201132258728649e-03
            1.56547269678219622135451061240019e-02 1.56547269678213585297754661951330e-02 1.59733808688937851776035170558998e-03
            1.56547269678226734501702566149106e-02 9.68690546064356228228575673711020e-01 1.59733808688943446259245195761878e-03
            9.68690546064355784139365823648404e-01 1.56547269678229059031160375070613e-02 1.59733808688946872338110249955889e-03
            3.34319867363665590431764940149151e-01 6.55493203809416224991934996069176e-01 4.05982765949562240254122613691834e-03
            3.34319867363664369186437852476956e-01 1.01869288269104425054267792916107e-02 4.05982765949340542593892777745168e-03
            6.55493203809410007742997095192550e-01 3.34319867363671197058039297189680e-01 4.05982765949597975557727735917979e-03
            6.55493203809432212203489598323358e-01 1.01869288269144011443989583654002e-02 4.05982765949478713318754330430238e-03
            1.01869288269114590533837017005681e-02 3.34319867363656542114114245123346e-01 4.05982765949380701442361640829404e-03
            1.01869288269143022651608276873958e-02 6.55493203809432323225792060839012e-01 4.05982765949477585748494945505627e-03
            2.92221537796962860245031379236025e-01 5.72337590532031481593833177612396e-01 1.34028711415783204768947811658109e-02
            2.92221537796955033172707771882415e-01 1.35440871671014539945332444403903e-01 1.34028711415832453568430793211519e-02
            5.72337590532020046296679538500030e-01 2.92221537796963026778485073009506e-01 1.34028711415786223187796011302453e-02
            5.72337590532020046296679538500030e-01 1.35440871671038687296118041558657e-01 1.34028711415840121046194610698876e-02
            1.35440871671019202882035870061372e-01 2.92221537796959529575957503766404e-01 1.34028711415833268888464502310853e-02
            1.35440871671036022760858941182960e-01 5.72337590532032480794555340253282e-01 1.34028711415840641463237403741005e-02
            3.19574885423182797339336502773222e-01 6.26001190286244124649783771019429e-01 9.22999660540936399388822763967255e-03
            3.19574885423193177924616747986875e-01 5.44239242905576528497313404386659e-02 9.22999660541005094438471445528194e-03
            6.26001190286247455318857646489050e-01 3.19574885423173693510534576489590e-01 9.22999660541028339733049534743259e-03
            6.26001190286228026415926706249593e-01 5.44239242905720371767941401230928e-02 9.22999660541144739678287578499294e-03
            5.44239242905613374023943151769345e-02 3.19574885423196675127144317229977e-01 9.22999660541019319170974455346368e-03
            5.44239242905700942865010460991471e-02 6.26001190286225028813760218326934e-01 9.22999660541093912280441458051428e-03
            1.90704224192300292317625576288265e-01 7.96427214974067321229256322112633e-01 4.23843426716296635403669057495790e-03
            1.90704224192293464446024131575541e-01 1.28685608336373685789411069890775e-02 4.23843426716443566482084293056687e-03
            7.96427214974062658292552896455163e-01 1.90704224192302679297128520374827e-01 4.23843426716371575457831255562269e-03
            7.96427214974071762121354822738795e-01 1.28685608336418667169143148498733e-02 4.23843426716566211431835853318262e-03
            1.28685608336382498184669032070815e-02 1.90704224192287385974964308843482e-01 4.23843426716460653508322664606567e-03
            1.28685608336411294594370247068582e-02 7.96427214974073316433589297957951e-01 4.23843426716543486554300557145325e-03
            1.80483211648735847321134428966616e-01 7.52351005937754502284064983541612e-01 9.14639838501133360659789417468346e-03
            1.80483211648742092325647945472156e-01 6.71657824135280245858581338325166e-02 9.14639838501276128401862308692216e-03
            7.52351005937751393659596033103298e-01 1.80483211648729463538742834316508e-01 9.14639838501189045283368272976077e-03
            7.52351005937707206783215951872990e-01 6.71657824135427627965100327855907e-02 9.14639838501379691393378124075753e-03
            6.71657824135295095091535699793894e-02 1.80483211648748614885917618266831e-01 9.14639838501270230342043987548095e-03
            6.71657824135391545716800010268344e-02 7.52351005937705763493283939169487e-01 9.14639838501353670541238471969336e-03
            8.07113136795677837964646528234880e-02 9.04625504095602384069252366316505e-01 3.33281600208312897212770131716297e-03
            8.07113136795645502719054320550640e-02 1.46631822248288815058181455697195e-02 3.33281600208281195141246655566647e-03
            9.04625504095601939980042516253889e-01 8.07113136795689078972770857944852e-02 3.33281600208304266963477147101003e-03
            9.04625504095612265054171530209715e-01 1.46631822248259914565071682091002e-02 3.33281600208207426025430741844957e-03
            1.46631822248279152648420264881679e-02 8.07113136795616359364657910191454e-02 3.33281600208249406333549380576642e-03
            1.46631822248264077901414026428029e-02 9.04625504095612598121078917756677e-01 3.33281600208215015440638140375995e-03
            ];
        
    case  18
        
        % ALG. DEG.:   18
        % PTS CARD.:   70
        % NEG. W.  :    3
        % OUT PTS. :    6
        % M.E.INF. : 1.1e-15
        
        xyw=[
            3.33333333333334980164153193982202e-01 3.33333333333333370340767487505218e-01 1.54049699688238728167988256245735e-02
            4.93344808630921449754680452315370e-01 4.93344808630920728109714445963618e-01 4.53621833970231994048738499714091e-03
            4.93344808630921671799285377346678e-01 1.33103827381572739629866930499702e-02 4.53621833970216901954497501492369e-03
            1.33103827381569617127610172246932e-02 4.93344808630922948555763696276699e-01 4.53621833970205973196598847607675e-03
            4.69210594241956047145691854893812e-01 4.69210594241956324701448011182947e-01 9.38065846979683863604559945770234e-03
            4.69210594241957601457926330112969e-01 6.15788115160862542518671602920222e-02 9.38065846979680220685260394475335e-03
            6.15788115160853521956596523523331e-02 4.69210594241957323902170173823833e-01 9.38065846979677965544741624626113e-03
            4.36281395887005984945972159039229e-01 4.36281395887005984945972159039229e-01 9.72054899273817191174273943943263e-03
            4.36281395887004486144888915077900e-01 1.27437208225988030108055681921542e-01 9.72054899273855875507788226741468e-03
            1.27437208225988030108055681921542e-01 4.36281395887005984945972159039229e-01 9.72054899273883978028099051016397e-03
            3.94846170673417984353648080286803e-01 3.94846170673417595775589461482014e-01 1.38769743054048492331142838906999e-02
            3.94846170673418095375950542802457e-01 2.10307658653166390516631167884043e-01 1.38769743054048873970307553804560e-02
            2.10307658653168028095592489989940e-01 3.94846170673415985952203755005030e-01 1.38769743054049377040115587078617e-02
            2.49794568803158312242373995104572e-01 2.49794568803156397107656516709540e-01 1.61281126757286744077823925636039e-02
            2.49794568803157840397588529413042e-01 5.00410862393686040050511110166553e-01 1.61281126757286535911006808419188e-02
            5.00410862393687150273535735323094e-01 2.49794568803157479575105526237166e-01 1.61281126757286327744189691202337e-02
            1.61432193743842294919588198354177e-01 1.61432193743842877786676126561360e-01 1.25370163084610065223900576825145e-02
            1.61432193743842239408436967096350e-01 6.77135612512314688515857596939895e-01 1.25370163084609839709848699840222e-02
            6.77135612512314799538160059455549e-01 1.61432193743842156141710120209609e-01 1.25370163084609839709848699840222e-02
            7.65982274853715949802790419198573e-02 7.65982274853715949802790419198573e-02 7.63596398591583763737489221057331e-03
            7.65982274853713035467350778162654e-02 8.46803545029257365150954228738556e-01 7.63596398591583503528967824536267e-03
            8.46803545029257587195559153769864e-01 7.65982274853712480355838465584384e-02 7.63596398591584804571574807141587e-03
            2.42524393534500198876813215065340e-02 2.42524393534499713154239941559354e-02 3.39696101148146637621616505953170e-03
            2.42524393534500233571282734601482e-02 9.51495121293100099002515435131500e-01 3.39696101148147071302485500154944e-03
            9.51495121293099876957910510100191e-01 2.42524393534500302960221773673766e-02 3.39696101148149326443004270004167e-03
            4.31463672169649994914131241330324e-02 4.31463672169649994914131241330324e-02 -1.11154936496017609989794827640708e-03
            4.31463672169649994914131241330324e-02 9.13707265566069959383810328290565e-01 -1.11154936496021773326137171977734e-03
            9.13707265566069959383810328290565e-01 4.31463672169649994914131241330324e-02 -1.11154936496026478763565759066978e-03
            3.58911494940944875153832072101068e-01 6.32657968856635499754759166535223e-01 3.16595703820280859747127166770042e-03
            3.58911494940944486575773453296279e-01 8.43053620242006744589513544951842e-03 3.16595703820291007879461631091544e-03
            6.32657968856636054866271479113493e-01 3.58911494940944431064622222038452e-01 3.16595703820275872417133733449646e-03
            6.32657968856636054866271479113493e-01 8.43053620242038836973819115883089e-03 3.16595703820300115177710509328790e-03
            8.43053620242074225332729042747815e-03 3.58911494940944653109227147069760e-01 3.16595703820311304144130559734549e-03
            8.43053620242064163936568377266667e-03 6.32657968856636943044691179238725e-01 3.16595703820306360182224025834330e-03
            2.94402476751956576617175187493558e-01 5.74410971510856294131031063443515e-01 1.36287690245692986318326589412209e-02
            2.94402476751956132527965337430942e-01 1.31186551737187878652335371043591e-01 1.36287690245692032220414802168307e-02
            5.74410971510856516175635988474824e-01 2.94402476751956798661780112524866e-01 1.36287690245692882234918030803783e-02
            5.74410971510854295729586738161743e-01 1.31186551737188572541725761766429e-01 1.36287690245692344470640477993584e-02
            1.31186551737189432964569846262748e-01 2.94402476751956299061419031204423e-01 1.36287690245692014873180042400236e-02
            1.31186551737189294186691768118180e-01 5.74410971510854184707284275646089e-01 1.36287690245691667928484847038817e-02
            3.25017801641814485513748422818026e-01 6.24779046792512171570876944315387e-01 8.83839282473228120273756047708957e-03
            3.25017801641814374491445960302372e-01 5.02031515656742102771126212701347e-02 8.83839282473231069303665208281018e-03
            6.24779046792512282593179406831041e-01 3.25017801641814707558353347849334e-01 8.83839282473225518188542082498316e-03
            6.24779046792512171570876944315387e-01 5.02031515656749943721237627869414e-02 8.83839282473231069303665208281018e-03
            5.02031515656757229559836730459210e-02 3.25017801641813042223816410114523e-01 8.83839282473233844861226771172369e-03
            5.02031515656756952004080574170075e-02 6.24779046792511727481667094252771e-01 8.83839282473235753057050345660173e-03
            1.84737559666046352591095569550816e-01 7.48933176523037147376271605025977e-01 9.18974231903505953122923699538660e-03
            1.84737559666046630146851725839952e-01 6.63292638109162224768766691340716e-02 9.18974231903503871454752527370147e-03
            7.48933176523037147376271605025977e-01 1.84737559666046380346671185179730e-01 9.18974231903506647012314090261498e-03
            7.48933176523037591465481455088593e-01 6.63292638109162085990888613196148e-02 9.18974231903499534646062585352411e-03
            6.63292638109160698212107831750473e-02 1.84737559666046574635700494582125e-01 9.18974231903499534646062585352411e-03
            6.63292638109162085990888613196148e-02 7.48933176523037591465481455088593e-01 9.18974231903498493811976999268154e-03
            2.18796800013320913436132286733482e-01 7.69207005420443290510945644200547e-01 4.05236640409595717110535773031188e-03
            2.18796800013321107725161596135877e-01 1.19961945662357839098577372283216e-02 4.05236640409594589540276388106577e-03
            7.69207005420443512555550569231855e-01 2.18796800013320663635951746073260e-01 4.05236640409595196693492979989060e-03
            7.69207005420443734600155494263163e-01 1.19961945662358012570924969963926e-02 4.05236640409594242595581192745158e-03
            1.19961945662357370723238858545301e-02 2.18796800013321274258615289909358e-01 4.05236640409594416067928790425867e-03
            1.19961945662358272779446366484990e-02 7.69207005420443956644760419294471e-01 4.05236640409594329331754991585512e-03
            1.01179597136407631308152588189841e-01 8.83962302273466948854263591783820e-01 3.81706453536226212111537314797260e-03
            1.01179597136407659063728203818755e-01 1.48581005901254580015002915160949e-02 3.81706453536226602424319409578857e-03
            8.83962302273467059876566054299474e-01 1.01179597136407548041425741303101e-01 3.81706453536225691694494521755132e-03
            8.83962302273467170898868516815128e-01 1.48581005901254319806481518639885e-02 3.81706453536225214645538628133181e-03
            1.48581005901254250417542479567601e-02 1.01179597136407728452667242891039e-01 3.81706453536226515688145610738502e-03
            1.48581005901254146334133920959175e-02 8.83962302273467281921170979330782e-01 3.81706453536224434019974438569989e-03
            2.08747552825863068315381809725295e-02 1.01434726000536290513309722882695e+00 2.30938303970509336952594142688966e-05
            2.08747552825862998926442770653011e-02 -3.52220152879488823671749742061365e-02 2.30938303970517061893073101908058e-05
            1.01434726000536290513309722882695e+00 2.08747552825861784620009586888045e-02 2.30938303970510861611899200429576e-05
            1.01434726000536290513309722882695e+00 -3.52220152879489933894774367217906e-02 2.30938303970512928372290500922404e-05
            -3.52220152879488823671749742061365e-02 2.08747552825862860148564692508444e-02 2.30938303970517265180980442940140e-05
            -3.52220152879489309394323015567352e-02 1.01434726000536290513309722882695e+00 2.30938303970514148099734547114892e-05
            ];
        
    case  19
        
        % ALG. DEG.:   19
        % PTS CARD.:   73
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 8.1e-16
        
        xyw=[
            3.33333333333333481363069950020872e-01 3.33333333333333259318465024989564e-01 1.64531656944593082658734317647031e-02
            4.89609987073006791558071881809155e-01 4.89609987073005792357349719168269e-01 5.16536594563603724400424610507798e-03
            4.89609987073006069913105875457404e-01 2.07800258539873258789576482286066e-02 5.16536594563602857038686622104251e-03
            2.07800258539873015928289845533072e-02 4.89609987073006736046920650551328e-01 5.16536594563600254953472656893609e-03
            4.54536892697891881098826161178295e-01 4.54536892697893601944514330170932e-01 1.11936236315081979325603001029776e-02
            4.54536892697893157855304480108316e-01 9.09262146042147667568400493109948e-02 1.11936236315081857894959682653280e-02
            9.09262146042150304348083977856732e-02 4.54536892697891659054221236146986e-01 1.11936236315081927283898721725564e-02
            4.01416680649430701599555959546706e-01 4.01416680649431811822580584703246e-01 1.51330629347340393786591761227101e-02
            4.01416680649431589777975659671938e-01 1.97166638701137625355741533894616e-01 1.51330629347340359092122241690959e-02
            1.97166638701137958422648921441578e-01 4.01416680649430424043799803257571e-01 1.51330629347340445828296040531313e-02
            2.55551654403098071899336218848475e-01 2.55551654403096961676311593691935e-01 1.52454839010988871250695808612363e-02
            2.55551654403097461276672675012378e-01 4.88896691193804799890898493686109e-01 1.52454839010988871250695808612363e-02
            4.88896691193804688868596031170455e-01 2.55551654403098127410487450106302e-01 1.52454839010988853903461048844292e-02
            1.77077942152129880204469714044535e-01 1.77077942152128908759323167032562e-01 1.20796063708204434727910836500087e-02
            1.77077942152129214070654938950611e-01 6.45844115695740961236026578262681e-01 1.20796063708204452075145596268158e-02
            6.45844115695740961236026578262681e-01 1.77077942152129991226772176560189e-01 1.20796063708204573505788914644654e-02
            1.10061053227951316513255619611300e-01 1.10061053227952621025309554170235e-01 8.02540179340041279210105074071180e-03
            1.10061053227951996524858202519681e-01 7.79877893544096201239312904363032e-01 8.02540179340043360878276246239693e-03
            7.79877893544096534306220291909995e-01 1.10061053227950997324136039878795e-01 8.02540179340044401712361832323950e-03
            5.55286242518394645983903501473833e-02 5.55286242518397074596769869003765e-02 4.04229013089202945246958265101966e-03
            5.55286242518397421541465064365184e-02 8.88942751496320737736311912158271e-01 4.04229013089202511566089270900193e-03
            8.88942751496320737736311912158271e-01 5.55286242518395062317537735907536e-02 4.04229013089202598302263069740548e-03
            1.26218637772287159415363078096561e-02 1.26218637772286621651085525286362e-02 1.03968101374239392346854327087158e-03
            1.26218637772286083886807972476163e-02 9.74756272445542637505866423452972e-01 1.03968101374239218874506729406448e-03
            9.74756272445542637505866423452972e-01 1.26218637772286916554076441343568e-02 1.03968101374239153822376380276182e-03
            3.95754787356943027454292405309388e-01 6.00633794794644737535804779327009e-01 1.94243845249071579381339169856346e-03
            3.95754787356943027454292405309388e-01 3.61141784841198434236053671497757e-03 1.94243845249066353526867789724975e-03
            6.00633794794644848558107241842663e-01 3.95754787356943082965443636567215e-01 1.94243845249067958146083068271537e-03
            6.00633794794644959580409704358317e-01 3.61141784841229182209665360403505e-03 1.94243845249072381690946809129628e-03
            3.61141784841220595328459275208388e-03 3.95754787356943027454292405309388e-01 1.94243845249070560231297033482178e-03
            3.61141784841207107853433555533229e-03 6.00633794794644959580409704358317e-01 1.94243845249067524465214074069763e-03
            3.07929983880435520493534795605228e-01 5.57603261588784704372301348485053e-01 1.27870803060109415605349880706854e-02
            3.07929983880436741738861883277423e-01 1.34466754530779386556105237104930e-01 1.27870803060109432952584640474925e-02
            5.57603261588783483126974260812858e-01 3.07929983880436963783466808308731e-01 1.27870803060109502341523679547208e-02
            5.57603261588784260283091498422436e-01 1.34466754530780219223373705972335e-01 1.27870803060109606424932238155634e-02
            1.34466754530780191467798090343422e-01 3.07929983880435298448929870573920e-01 1.27870803060109467647054160011066e-02
            1.34466754530779913912041934054287e-01 5.57603261588783261082369335781550e-01 1.27870803060109502341523679547208e-02
            2.64566948406519730774277832097141e-01 7.20987025817365734248198805289576e-01 4.44045178666900661884620049590922e-03
            2.64566948406520674463848763480200e-01 1.44460257761147691651926194822408e-02 4.44045178666902656816617422919080e-03
            7.20987025817364624025174180133035e-01 2.64566948406520563441546300964546e-01 4.44045178666903263969834014801563e-03
            7.20987025817365179136686492711306e-01 1.44460257761147743693630474126621e-02 4.44045178666903697650703009003337e-03
            1.44460257761147275318291960388706e-02 2.64566948406520341396941375933238e-01 4.44045178666903090497486417120854e-03
            1.44460257761148559013664183225956e-02 7.20987025817364068913661867554765e-01 4.44045178666905258901831388129722e-03
            3.58539352205950012653090652747778e-01 5.94527068955871595967721532360883e-01 8.06227338086568795505471030082845e-03
            3.58539352205951067364964046646492e-01 4.69335788381783636236121992624248e-02 8.06227338086568968977818627763554e-03
            5.94527068955871040856209219782613e-01 3.58539352205950401231149271552567e-01 8.06227338086569662867209018486392e-03
            5.94527068955870818811604294751305e-01 4.69335788381786758738378750877018e-02 8.06227338086570877173642202251358e-03
            4.69335788381785093403841813142208e-02 3.58539352205951067364964046646492e-01 8.06227338086569142450166225444264e-03
            4.69335788381786966905195868093870e-02 5.94527068955869819610882132110419e-01 8.06227338086571224118337397612777e-03
            1.57807405968594077760869254234422e-01 8.39331473680839379447604642336955e-01 1.24597090874531933601954136747736e-03
            1.57807405968594993694864569988567e-01 2.86112035056767339755157131264696e-03 1.24597090874547025696195134969457e-03
            8.39331473680837936157672629633453e-01 1.57807405968594993694864569988567e-01 1.24597090874539869961856730640193e-03
            8.39331473680839046380697254789993e-01 2.86112035056589183654174313176100e-03 1.24597090874522587779227311699515e-03
            2.86112035056642656505321298254785e-03 1.57807405968594993694864569988567e-01 1.24597090874530133826347810810375e-03
            2.86112035056694091056384010585134e-03 8.39331473680837381046160317055183e-01 1.24597090874538113554337304123010e-03
            7.50505969759109881067615788197145e-02 7.01087978926172805671512833214365e-01 9.12142005947530302989445516459455e-03
            7.50505969759108076955200772317767e-02 2.23861424097916178466149972337007e-01 9.12142005947528394793621941971651e-03
            7.01087978926173249760722683276981e-01 7.50505969759110713734884257064550e-02 9.12142005947529609100055125736617e-03
            7.01087978926173693849932533339597e-01 2.23861424097915318043305887840688e-01 9.12142005947528915210664735013779e-03
            2.23861424097915789888091353532218e-01 7.50505969759107660621566537884064e-02 9.12142005947527527431883953568104e-03
            2.23861424097915373554457119098515e-01 7.01087978926173804872234995855251e-01 9.12142005947527180487188758206685e-03
            1.42421601113383283765401188247779e-01 8.22931324069856917624576908565359e-01 5.12928186809924555739081952765446e-03
            1.42421601113382922942918185071903e-01 3.46470748167610753665002221168834e-02 5.12928186809922300598563182916223e-03
            8.22931324069856362513064595987089e-01 1.42421601113383228254249956989952e-01 5.12928186809924122058212958563672e-03
            8.22931324069857250691484296112321e-01 3.46470748167589451260717225977714e-02 5.12928186809929066020119492463891e-03
            3.46470748167595626876291703410971e-02 1.42421601113383672343459807052568e-01 5.12928186809926116990210331891831e-03
            3.46470748167603259659586001362186e-02 8.22931324069856140468459670955781e-01 5.12928186809927331296643515656797e-03
            6.54946280829374338772197461366886e-02 9.24344252620784345708671025931835e-01 1.89996442765094056444319381427022e-03
            6.54946280829379334775808274571318e-02 1.01611192962782828641543630965316e-02 1.89996442765096203164620902725801e-03
            9.24344252620783790597158713353565e-01 6.54946280829379057220052118282183e-02 1.89996442765096051376316754755180e-03
            9.24344252620784123664066100900527e-01 1.01611192962782394960674636763542e-02 1.89996442765094685281579423019593e-03
            1.01611192962781649029579966736492e-02 6.54946280829377808219149414981075e-02 1.89996442765094772017753221859948e-03
            1.01611192962782915377717429805671e-02 9.24344252620783568552553788322257e-01 1.89996442765097265682749938520146e-03
            ];
        
    case  20
        
        % ALG. DEG.:   20
        % PTS CARD.:   79
        % NEG. W.  :    3
        % OUT PTS. :    9
        % M.E.INF. : 9.9e-16
        
        xyw=[
            3.33333333333333425851918718763045e-01 3.33333333333333314829616256247391e-01 1.65285277708119432427125161666481e-02
            5.00950464352196256356819503707811e-01 5.00950464352196478401424428739119e-01 4.33509592831736012118887257926758e-04
            5.00950464352200031115103229240049e-01 -1.90092870439298304054143162744595e-03 4.33509592831716442269673894571724e-04
            -1.90092870439260855711105513421444e-03 5.00950464352200031115103229240049e-01 4.33509592831740945238772066971933e-04
            4.88212957934727465048041494810604e-01 4.88212957934727409536890263552777e-01 5.83002635822355442379860690493842e-03
            4.88212957934727798114948882357567e-01 2.35740841305451219456212896830039e-02 5.83002635822356483213946276578099e-03
            2.35740841305452641929463197811856e-02 4.88212957934727742603797651099740e-01 5.83002635822355355643686891653488e-03
            4.55136681950281951625214560408494e-01 4.55136681950281951625214560408494e-01 1.14384681782103564406938289721438e-02
            4.55136681950282062647517022924148e-01 8.97266360994361106273586869974679e-02 1.14384681782103616448642569025651e-02
            8.97266360994361522607221104408382e-02 4.55136681950282007136365791666321e-01 1.14384681782103581754173049489509e-02
            4.01996259318289195228146581939654e-01 4.01996259318289139716995350681827e-01 1.52244913369689836846188057961626e-02
            4.01996259318289195228146581939654e-01 1.96007481363421748321584914265259e-01 1.52244913369689888887892337265839e-02
            1.96007481363421831588311761151999e-01 4.01996259318289195228146581939654e-01 1.52244913369689854193422817729697e-02
            2.55892909759421249837885170563823e-01 2.55892909759421249837885170563823e-01 1.53124458626773228553386019257232e-02
            2.55892909759421194326733939305996e-01 4.88214180481157666857683352645836e-01 1.53124458626773245900620779025303e-02
            4.88214180481157777879985815161490e-01 2.55892909759421194326733939305996e-01 1.53124458626773193858916499721090e-02
            1.76488255995106035900477081668214e-01 1.76488255995106063656052697297127e-01 1.21840288384002098931446766982845e-02
            1.76488255995105980389325850410387e-01 6.47023488009788039221348299179226e-01 1.21840288384001977500803448606348e-02
            6.47023488009788150243650761694880e-01 1.76488255995106008144901466039300e-01 1.21840288384002046889742487678632e-02
            1.04170855336758461096025030201417e-01 1.04170855336758461096025030201417e-01 7.99871601601196031972040856317108e-03
            1.04170855336758405584873798943590e-01 7.91658289326483188830252402112819e-01 7.99871601601195511554998063274979e-03
            7.91658289326483299852554864628473e-01 1.04170855336758363951510375500220e-01 7.99871601601195164610302867913560e-03
            5.30689638409298294896565550970990e-02 5.30689638409297947951870355609572e-02 3.84915090780120483504589401491103e-03
            5.30689638409295935672638222513342e-02 8.93862072318140299387323466362432e-01 3.84915090780115366070335269910174e-03
            8.93862072318140521431928391393740e-01 5.30689638409294686671735519212234e-02 3.84915090780119225830069318305959e-03
            4.16187151960289991592389924335293e-02 4.16187151960289991592389924335293e-02 -3.16030248743842656893726639211195e-04
            4.16187151960239615222647557857272e-02 9.16762569607942001681522015132941e-01 -3.16030248743797011982264999474523e-04
            9.16762569607942001681522015132941e-01 4.16187151960243986725807019411150e-02 -3.16030248743838049034493575817351e-04
            1.15819214068221820745918648754014e-02 1.15819214068222063607205285507007e-02 8.75567150596384051290665961886361e-04
            1.15819214068221699315275330377517e-02 9.76836157186355635850816270249197e-01 8.75567150596381232365017499574833e-04
            9.76836157186355635850816270249197e-01 1.15819214068222028912735765970865e-02 8.75567150596381882886320990877493e-04
            3.44855770229001101778720794754918e-01 6.06402646106159570749127851740923e-01 8.23291959478787520321230175568417e-03
            3.44855770229001268312174488528399e-01 4.87415836648393274721513535041595e-02 8.23291959478788387682968163971964e-03
            6.06402646106159681771430314256577e-01 3.44855770229001101778720794754918e-01 8.23291959478786999904187382526288e-03
            6.06402646106159792793732776772231e-01 4.87415836648393829833025847619865e-02 8.23291959478787173376534980206998e-03
            4.87415836648393344110452574113879e-02 3.44855770229001268312174488528399e-01 8.23291959478787867265925370929835e-03
            4.87415836648393899221964886692149e-02 6.06402646106159792793732776772231e-01 8.23291959478786999904187382526288e-03
            3.77843269594853925497091040597297e-01 6.15842614456540871081813293130836e-01 2.41951677024240157168333276160865e-03
            3.77843269594855035720115665753838e-01 6.31411594860513576688010317639055e-03 2.41951677024240157168333276160865e-03
            6.15842614456540871081813293130836e-01 3.77843269594853981008242271855124e-01 2.41951677024239983695985678480156e-03
            6.15842614456541537215628068224760e-01 6.31411594860525893224689752969425e-03 2.41951677024239246438508388337141e-03
            6.31411594860514010368879311840828e-03 3.77843269594855035720115665753838e-01 2.41951677024240417376854672681930e-03
            6.31411594860526326905558747171199e-03 6.15842614456541537215628068224760e-01 2.41951677024239723487464281959092e-03
            3.06635479062356774981878970720572e-01 5.59048000390295452000088971544756e-01 1.29024532673249966996387172457617e-02
            3.06635479062356830493030201978399e-01 1.34316520547347800773607673363585e-01 1.29024532673250019038091451761829e-02
            5.59048000390295563022391434060410e-01 3.06635479062356719470727739462745e-01 1.29024532673250001690856691993758e-02
            5.59048000390295563022391434060410e-01 1.34316520547347773018032057734672e-01 1.29024532673249966996387172457617e-02
            1.34316520547347800773607673363585e-01 3.06635479062356830493030201978399e-01 1.29024532673250071079795731066042e-02
            1.34316520547347800773607673363585e-01 5.59048000390295563022391434060410e-01 1.29024532673249966996387172457617e-02
            2.49419362774742109367664966157463e-01 7.36606743262865681920459337561624e-01 4.23554552722032063433177384581541e-03
            2.49419362774742525701299200591166e-01 1.39738939623921532007244650230859e-02 4.23554552722032670586393976464024e-03
            7.36606743262865792942761800077278e-01 2.49419362774742109367664966157463e-01 4.23554552722032063433177384581541e-03
            7.36606743262866014987366725108586e-01 1.39738939623921601396183689303143e-02 4.23554552722030849126744200816574e-03
            1.39738939623921306493192773245937e-02 2.49419362774742525701299200591166e-01 4.23554552722032930794915372985088e-03
            1.39738939623921497312775130694718e-02 7.36606743262865903965064262592932e-01 4.23554552722031022599091798497284e-03
            2.12775724802801630142212729879247e-01 7.11675142287434225885078831197461e-01 9.17745705313986574747087132664092e-03
            2.12775724802801741164515192394902e-01 7.55491329097641856060718623666617e-02 9.17745705313987268636477523386930e-03
            7.11675142287434336907381293713115e-01 2.12775724802801602386637114250334e-01 9.17745705313986574747087132664092e-03
            7.11675142287434447929683756228769e-01 7.55491329097641439727084389232914e-02 9.17745705313986401274739534983382e-03
            7.55491329097641856060718623666617e-02 2.12775724802801713408939576765988e-01 9.17745705313987442108825121067639e-03
            7.55491329097641439727084389232914e-02 7.11675142287434336907381293713115e-01 9.17745705313986574747087132664092e-03
            1.46965436053239223435085136770795e-01 8.61402717154987396241949682007544e-01 3.52202338954109598831432315080292e-04
            1.46965436053239223435085136770795e-01 -8.36815320822662141175829475514547e-03 3.52202338954107647267521841172311e-04
            8.61402717154987396241949682007544e-01 1.46965436053239167923933905512968e-01 3.52202338954106942536109725594429e-04
            8.61402717154987951353461994585814e-01 -8.36815320822675498546594496929174e-03 3.52202338954102388886985286475806e-04
            -8.36815320822661620758786682472419e-03 1.46965436053239223435085136770795e-01 3.52202338954108189368608083924528e-04
            -8.36815320822669427014428578104344e-03 8.61402717154987729308857069554506e-01 3.52202338954105099392416500236891e-04
            1.37726978828923041220377854187973e-01 8.35586957912362815470430632558418e-01 5.05634246373094613236087724317258e-03
            1.37726978828923263264982779219281e-01 2.66860632587140808591463780885533e-02 5.05634246373095393861651913880451e-03
            8.35586957912362926492733095074072e-01 1.37726978828922985709226622930146e-01 5.05634246373095133653130517359386e-03
            8.35586957912362926492733095074072e-01 2.66860632587140218785481948771121e-02 5.05634246373093832610523534754066e-03
            2.66860632587141086147219937174668e-02 1.37726978828923235509407163590367e-01 5.05634246373095740806347109241869e-03
            2.66860632587140565730177144132540e-02 8.35586957912362815470430632558418e-01 5.05634246373093572402002138233001e-03
            5.96961091490066306231021542316739e-02 9.29756171556852639703549812111305e-01 1.78695469297513289189116569843918e-03
            5.96961091490065751119509229738469e-02 1.05477192941408511039913520335176e-02 1.78695469297516411691373328096688e-03
            9.29756171556852750725852274626959e-01 5.96961091490065196007996917160199e-02 1.78695469297513571081681416075071e-03
            9.29756171556852639703549812111305e-01 1.05477192941409447790590547811007e-02 1.78695469297518081362718955773516e-03
            1.05477192941408910026312995000808e-02 5.96961091490064918452240760871064e-02 1.78695469297516541795634026357220e-03
            1.05477192941409551873999106419433e-02 9.29756171556852639703549812111305e-01 1.78695469297518276519110003164315e-03
            ];
        
        
    otherwise
        xyw=[];
end




pointset_stats=[
    1    1 1.92e-16    0    0
    2    3 3.68e-16    0    0
    3    4 2.22e-16    1    0
    4    6 2.01e-16    0    0
    5    7 3.15e-16    0    0
    6   12 3.12e-16    0    0
    7   13 3.75e-16    1    0
    8   16 6.11e-16    0    0
    9   19 3.19e-16    0    0
    10   25 5.07e-16    0    0
    11   27 8.88e-16    0    3
    12   33 3.89e-16    0    0
    13   37 4.79e-16    0    0
    14   42 8.47e-16    0    0
    15   48 8.47e-16    0    9
    16   52 9.02e-16    0    6
    17   61 5.64e-16    0    0
    18   70 1.11e-15    3    6
    19   73 8.10e-16    0    0
    20   79 9.91e-16    3    9
    ];




