
function [xyw,pointset_stats]=set_dunavant_standard(deg)

%--------------------------------------------------------------------
% CUBATURE RULE ON THE SIMPLEX [0 0; 0 1; 1 0] WITH ADE EQUAL TO deg.
% THE CUBATURE RULE IS DESCRIBED IN STANDARD FORM.
%--------------------------------------------------------------------
% INPUT:
% deg: algebraic degree of precision (ADE) of the rule.
%--------------------------------------------------------------------
% OUTPUT:
% xyw: MATRIX OF NODES xyw(:,1:2) AND WEIGHTS xyw(:,3).
%      REMARK: The sum of the weights makes 1/2.
%
% pointset_stats:
%    statistics of the pointset stored in a matrix. Each rows contains
%    degree, cardinality of the full pointset, moment error w.r.t. orthn.
%    Dubiner basis,number of points with negative weights, number of
%    points outside the domain.
%--------------------------------------------------------------------
% SOURCE:
%  D.A. Dunavant,
%  "High Degree Efficient Symmetrical Gaussian Quadrature Rules for the
%  Triangle"
%  International Journal for Numerical Methods in Engineering,
%  21(6), June 1985, pp. 1129-1148
%--------------------------------------------------------------------
% | DEG | CARD | W- | PT.O|   M.E.   |Q.|
%--------------------------------------------------------------------
% |   1 |   1 |   0 |   0 | 1.92e-16 |PI|
% |   2 |   3 |   0 |   0 | 3.68e-16 |PI|
% |   3 |   4 |   1 |   0 | 2.22e-16 |NI|
% |   4 |   6 |   0 |   0 | 2.24e-15 |PI|
% |   5 |   7 |   0 |   0 | 1.79e-13 |PI|
% |   6 |  12 |   0 |   0 | 2.75e-15 |PI|
% |   7 |  13 |   1 |   0 | 4.16e-15 |NI|
% |   8 |  16 |   0 |   0 | 3.87e-15 |PI|
% |   9 |  19 |   0 |   0 | 3.43e-15 |PI|
% |  10 |  25 |   0 |   0 | 1.10e-14 |PI|
% |  11 |  27 |   0 |   3 | 1.20e-13 |P0|
% |  12 |  33 |   0 |   0 | 4.54e-15 |PI|
% |  13 |  37 |   0 |   0 | 9.07e-15 |PI|
% |  14 |  42 |   0 |   0 | 6.71e-15 |PI|
% |  15 |  48 |   0 |   9 | 3.60e-14 |P0|
% |  16 |  52 |   0 |   6 | 1.98e-14 |P0|
% |  17 |  61 |   0 |   0 | 1.10e-14 |PI|
% |  18 |  70 |   3 |   6 | 5.08e-13 |N0|
% |  19 |  73 |   0 |   0 | 1.65e-14 |PI|
% |  20 |  79 |   3 |   9 | 4.74e-14 |N0|
%--------------------------------------------------------------------
% LEGEND:
%--------------------------------------------------------------------
% DEG  : algebraic degree of precision
% CARD : cardinality of the pointset
% W-   : negative weights
% PT.O : points outside the simplex [0 0; 1 0; 0 1].
% M.E. : maximum moment error in inf norm.
% Q.: quality of the rule.
%        PI: pos. weights, points in simplex
%        NI: some neg. weights, points in simplex
%        P0: pos. weights, some points out of the simplex
%        N0: some neg. weights, some points out of the simplex
%--------------------------------------------------------------------


switch deg
    
    case   1
        
        % ALG. DEG.:    1
        % PTS CARD.:    1
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.9e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 5.00000000000000000000000000000000e-01
            ];
        
    case   2
        
        % ALG. DEG.:    2
        % PTS CARD.:    3
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.7e-16
        
        xyw=[
            1.66666666666666657414808128123695e-01 1.66666666666666657414808128123695e-01 1.66666666666666657414808128123695e-01
            1.66666666666666657414808128123695e-01 6.66666666666666740681534975010436e-01 1.66666666666666657414808128123695e-01
            6.66666666666666740681534975010436e-01 1.66666666666666657414808128123695e-01 1.66666666666666657414808128123695e-01
            ];
        
    case   3
        
        % ALG. DEG.:    3
        % PTS CARD.:    4
        % NEG. W.  :    1
        % OUT PTS. :    0
        % M.E.INF. : 2.2e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 -2.81250000000000000000000000000000e-01
            2.00000000000000011102230246251565e-01 2.00000000000000011102230246251565e-01 2.60416666666666685170383743752609e-01
            2.00000000000000011102230246251565e-01 5.99999999999999977795539507496869e-01 2.60416666666666685170383743752609e-01
            5.99999999999999977795539507496869e-01 2.00000000000000011102230246251565e-01 2.60416666666666685170383743752609e-01
            ];
        
    case   4
        
        % ALG. DEG.:    4
        % PTS CARD.:    6
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 2.2e-15
        
        xyw=[
            4.45948490915965001235576892213430e-01 4.45948490915965001235576892213430e-01 1.11690794839005624883299105931656e-01
            4.45948490915965001235576892213430e-01 1.08103018168069997528846215573139e-01 1.11690794839005624883299105931656e-01
            1.08103018168069997528846215573139e-01 4.45948490915965001235576892213430e-01 1.11690794839005624883299105931656e-01
            9.15762135097710067155318824916321e-02 9.15762135097710067155318824916321e-02 5.49758718276610602870846378209535e-02
            9.15762135097710067155318824916321e-02 8.16847572980457958813360619387822e-01 5.49758718276610602870846378209535e-02
            8.16847572980457958813360619387822e-01 9.15762135097710067155318824916321e-02 5.49758718276610602870846378209535e-02
            ];
        
    case   5
        
        % ALG. DEG.:    5
        % PTS CARD.:    7
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.8e-13
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 1.12500000000000127675647831893002e-01
            4.70142064105100010440452251714305e-01 4.70142064105100010440452251714305e-01 6.61970763942530682211184966945439e-02
            4.70142064105100010440452251714305e-01 5.97158717897999791190954965713900e-02 6.61970763942530682211184966945439e-02
            5.97158717897999791190954965713900e-02 4.70142064105100010440452251714305e-01 6.61970763942530682211184966945439e-02
            1.01286507323455995943639607048681e-01 1.01286507323455995943639607048681e-01 6.29695902724135697647867004889122e-02
            1.01286507323455995943639607048681e-01 7.97426985353087980357145170273725e-01 6.29695902724135697647867004889122e-02
            7.97426985353087980357145170273725e-01 1.01286507323455995943639607048681e-01 6.29695902724135697647867004889122e-02
            ];
        
    case   6
        
        % ALG. DEG.:    6
        % PTS CARD.:   12
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 2.7e-15
        
        xyw=[
            2.49286745170910012392440080475353e-01 2.49286745170910012392440080475353e-01 5.83931378631893788222484431571502e-02
            2.49286745170910012392440080475353e-01 5.01426509658180030726271070307121e-01 5.83931378631893788222484431571502e-02
            5.01426509658180030726271070307121e-01 2.49286745170910012392440080475353e-01 5.83931378631893788222484431571502e-02
            6.30890144915020045779385782225290e-02 6.30890144915020045779385782225290e-02 2.54224531851034475649075972114588e-02
            6.30890144915020045779385782225290e-02 8.73821971016995990844122843554942e-01 2.54224531851034475649075972114588e-02
            8.73821971016995990844122843554942e-01 6.30890144915020045779385782225290e-02 2.54224531851034475649075972114588e-02
            5.31450498448170008392033025756973e-02 3.10352451033784004774673803694895e-01 4.14255378091869172485495198543504e-02
            5.31450498448170008392033025756973e-02 6.36502499121399001325016797636636e-01 4.14255378091869172485495198543504e-02
            3.10352451033784004774673803694895e-01 5.31450498448170008392033025756973e-02 4.14255378091869172485495198543504e-02
            3.10352451033784004774673803694895e-01 6.36502499121399001325016797636636e-01 4.14255378091869172485495198543504e-02
            6.36502499121399001325016797636636e-01 5.31450498448170008392033025756973e-02 4.14255378091869172485495198543504e-02
            6.36502499121399001325016797636636e-01 3.10352451033784004774673803694895e-01 4.14255378091869172485495198543504e-02
            ];
        
    case   7
        
        % ALG. DEG.:    7
        % PTS CARD.:   13
        % NEG. W.  :    1
        % OUT PTS. :    0
        % M.E.INF. : 4.2e-15
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 -7.47850222338411474432007253199117e-02
            2.60345966079039981000420311829657e-01 2.60345966079039981000420311829657e-01 8.78076287166041774545632847548404e-02
            2.60345966079039981000420311829657e-01 4.79308067841920037999159376340685e-01 8.78076287166041774545632847548404e-02
            4.79308067841920037999159376340685e-01 2.60345966079039981000420311829657e-01 8.78076287166041774545632847548404e-02
            6.51301029022160055115264754022064e-02 6.51301029022160055115264754022064e-02 2.66736178044190523628298450375951e-02
            6.51301029022160055115264754022064e-02 8.69739794195568016732522664824501e-01 2.66736178044190523628298450375951e-02
            8.69739794195568016732522664824501e-01 6.51301029022160055115264754022064e-02 2.66736178044190523628298450375951e-02
            4.86903154253160025399793653377856e-02 3.12865496004874010793628258397803e-01 3.85568804451285745971134133469604e-02
            4.86903154253160025399793653377856e-02 6.38444188569810000544180184078868e-01 3.85568804451285745971134133469604e-02
            3.12865496004874010793628258397803e-01 4.86903154253160025399793653377856e-02 3.85568804451285745971134133469604e-02
            3.12865496004874010793628258397803e-01 6.38444188569810000544180184078868e-01 3.85568804451285745971134133469604e-02
            6.38444188569810000544180184078868e-01 4.86903154253160025399793653377856e-02 3.85568804451285745971134133469604e-02
            6.38444188569810000544180184078868e-01 3.12865496004874010793628258397803e-01 3.85568804451285745971134133469604e-02
            ];
        
    case   8
        
        % ALG. DEG.:    8
        % PTS CARD.:   16
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.9e-15
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 7.21578038388935028013548844683100e-02
            4.59292588292723014120610969257541e-01 4.59292588292723014120610969257541e-01 4.75458171336424970099265863154869e-02
            4.59292588292723014120610969257541e-01 8.14148234145539717587780614849180e-02 4.75458171336424970099265863154869e-02
            8.14148234145539717587780614849180e-02 4.59292588292723014120610969257541e-01 4.75458171336424970099265863154869e-02
            1.70569307751759990932072241776041e-01 1.70569307751759990932072241776041e-01 5.16086852673589974171974859018519e-02
            1.70569307751759990932072241776041e-01 6.58861384496480018135855516447918e-01 5.16086852673589974171974859018519e-02
            6.58861384496480018135855516447918e-01 1.70569307751759990932072241776041e-01 5.16086852673589974171974859018519e-02
            5.05472283170309982791579272998206e-02 5.05472283170309982791579272998206e-02 1.62292488115990014840850363952995e-02
            5.05472283170309982791579272998206e-02 8.98905543365937975686108529771445e-01 1.62292488115990014840850363952995e-02
            8.98905543365937975686108529771445e-01 5.05472283170309982791579272998206e-02 1.62292488115990014840850363952995e-02
            8.39477740995800067458354476457316e-03 2.63112829634638001330415590928169e-01 1.36151570872174998427794889721554e-02
            8.39477740995800067458354476457316e-03 7.28492392955404022281129527982557e-01 1.36151570872174998427794889721554e-02
            2.63112829634638001330415590928169e-01 8.39477740995800067458354476457316e-03 1.36151570872174998427794889721554e-02
            2.63112829634638001330415590928169e-01 7.28492392955404022281129527982557e-01 1.36151570872174998427794889721554e-02
            7.28492392955404022281129527982557e-01 8.39477740995800067458354476457316e-03 1.36151570872174998427794889721554e-02
            7.28492392955404022281129527982557e-01 2.63112829634638001330415590928169e-01 1.36151570872174998427794889721554e-02
            ];
        
    case   9
        
        % ALG. DEG.:    9
        % PTS CARD.:   19
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.4e-15
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 4.85678981413997096217016746777517e-02
            4.89682519198738008814331124085584e-01 4.89682519198738008814331124085584e-01 1.56673501135695669717051714542322e-02
            4.89682519198738008814331124085584e-01 2.06349616025239823713377518288326e-02 1.56673501135695669717051714542322e-02
            2.06349616025239823713377518288326e-02 4.89682519198738008814331124085584e-01 1.56673501135695669717051714542322e-02
            4.37089591492937024064246998023009e-01 4.37089591492937024064246998023009e-01 3.89137705023871668941204404745804e-02
            4.37089591492937024064246998023009e-01 1.25820817014125951871506003953982e-01 3.89137705023871668941204404745804e-02
            1.25820817014125951871506003953982e-01 4.37089591492937024064246998023009e-01 3.89137705023871668941204404745804e-02
            1.88203535619032996661914580727171e-01 1.88203535619032996661914580727171e-01 3.98238694636051659969666616234463e-02
            1.88203535619032996661914580727171e-01 6.23592928761933951165019607287832e-01 3.98238694636051659969666616234463e-02
            6.23592928761933951165019607287832e-01 1.88203535619032996661914580727171e-01 3.98238694636051659969666616234463e-02
            4.47295133944529965663861048597028e-02 4.47295133944529965663861048597028e-02 1.27888378293490537901400116993500e-02
            4.47295133944529965663861048597028e-02 9.10540973211094062378379021538422e-01 1.27888378293490537901400116993500e-02
            9.10540973211094062378379021538422e-01 4.47295133944529965663861048597028e-02 1.27888378293490537901400116993500e-02
            3.68384120547360013886439844554843e-02 2.21962989160766011043079970477265e-01 2.16417696886445909409957266689162e-02
            3.68384120547360013886439844554843e-02 7.41198598784498008384957756788936e-01 2.16417696886445909409957266689162e-02
            2.21962989160766011043079970477265e-01 3.68384120547360013886439844554843e-02 2.16417696886445909409957266689162e-02
            2.21962989160766011043079970477265e-01 7.41198598784498008384957756788936e-01 2.16417696886445909409957266689162e-02
            7.41198598784498008384957756788936e-01 3.68384120547360013886439844554843e-02 2.16417696886445909409957266689162e-02
            7.41198598784498008384957756788936e-01 2.21962989160766011043079970477265e-01 2.16417696886445909409957266689162e-02
            ];
        
    case  10
        
        % ALG. DEG.:   10
        % PTS CARD.:   25
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.1e-14
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 4.54089951913770398239655889938149e-02
            4.85577633383657003562916543160100e-01 4.85577633383657003562916543160100e-01 1.83629788782335157115710444486467e-02
            4.85577633383657003562916543160100e-01 2.88447332326859928741669136798009e-02 1.83629788782335157115710444486467e-02
            2.88447332326859928741669136798009e-02 4.85577633383657003562916543160100e-01 1.83629788782335157115710444486467e-02
            1.09481575485036994321852432676678e-01 1.09481575485036994321852432676678e-01 2.26605297177640205297244335724827e-02
            1.09481575485036994321852432676678e-01 7.81036849029925983600719519017730e-01 2.26605297177640205297244335724827e-02
            7.81036849029925983600719519017730e-01 1.09481575485036994321852432676678e-01 2.26605297177640205297244335724827e-02
            1.41707219414879986851829585248197e-01 3.07939838764121009972996034775861e-01 3.63789584227100365465545905863110e-02
            1.41707219414879986851829585248197e-01 5.50352941820999030930749995604856e-01 3.63789584227100365465545905863110e-02
            3.07939838764121009972996034775861e-01 1.41707219414879986851829585248197e-01 3.63789584227100365465545905863110e-02
            3.07939838764121009972996034775861e-01 5.50352941820999030930749995604856e-01 3.63789584227100365465545905863110e-02
            5.50352941820999030930749995604856e-01 1.41707219414879986851829585248197e-01 3.63789584227100365465545905863110e-02
            5.50352941820999030930749995604856e-01 3.07939838764121009972996034775861e-01 3.63789584227100365465545905863110e-02
            2.50035347626859985603609004556347e-02 2.46672560639903004631179328498547e-01 1.41636212655285118749981521091286e-02
            2.50035347626859985603609004556347e-02 7.28323904597411031502929290581960e-01 1.41636212655285118749981521091286e-02
            2.46672560639903004631179328498547e-01 2.50035347626859985603609004556347e-02 1.41636212655285118749981521091286e-02
            2.46672560639903004631179328498547e-01 7.28323904597411031502929290581960e-01 1.41636212655285118749981521091286e-02
            7.28323904597411031502929290581960e-01 2.50035347626859985603609004556347e-02 1.41636212655285118749981521091286e-02
            7.28323904597411031502929290581960e-01 2.46672560639903004631179328498547e-01 1.41636212655285118749981521091286e-02
            9.54081540029899990984496582768770e-03 6.68032510121999989483043691507191e-02 4.71083348186650437139366687233633e-03
            9.54081540029899990984496582768770e-03 9.23655933587500976855722001346294e-01 4.71083348186650437139366687233633e-03
            6.68032510121999989483043691507191e-02 9.54081540029899990984496582768770e-03 4.71083348186650437139366687233633e-03
            6.68032510121999989483043691507191e-02 9.23655933587500976855722001346294e-01 4.71083348186650437139366687233633e-03
            9.23655933587500976855722001346294e-01 9.54081540029899990984496582768770e-03 4.71083348186650437139366687233633e-03
            9.23655933587500976855722001346294e-01 6.68032510121999989483043691507191e-02 4.71083348186650437139366687233633e-03
            ];
        
    case  11
        
        % ALG. DEG.:   11
        % PTS CARD.:   27
        % NEG. W.  :    0
        % OUT PTS. :    3
        % M.E.INF. : 1.2e-13
        
        xyw=[
            5.34611048270758004719027667306364e-01 5.34611048270758004719027667306364e-01 4.63503164480498974428512859446982e-04
            5.34611048270758004719027667306364e-01 -6.92220965415160094380553346127272e-02 4.63503164480498974428512859446982e-04
            -6.92220965415160094380553346127272e-02 5.34611048270758004719027667306364e-01 4.63503164480498974428512859446982e-04
            3.98969302965854977216508814308327e-01 3.98969302965854977216508814308327e-01 3.85747674574064142172247215967218e-02
            3.98969302965854977216508814308327e-01 2.02061394068290045566982371383347e-01 3.85747674574064142172247215967218e-02
            2.02061394068290045566982371383347e-01 3.98969302965854977216508814308327e-01 3.85747674574064142172247215967218e-02
            2.03309900431281997956745044575655e-01 2.03309900431281997956745044575655e-01 2.96614886903869348633300262463308e-02
            2.03309900431281997956745044575655e-01 5.93380199137436004086509910848690e-01 2.96614886903869348633300262463308e-02
            5.93380199137436004086509910848690e-01 2.03309900431281997956745044575655e-01 2.96614886903869348633300262463308e-02
            1.19350912282581003709402978074650e-01 1.19350912282581003709402978074650e-01 1.80922702517089598051036603010289e-02
            1.19350912282581003709402978074650e-01 7.61298175434838020336769659479614e-01 1.80922702517089598051036603010289e-02
            7.61298175434838020336769659479614e-01 1.19350912282581003709402978074650e-01 1.80922702517089598051036603010289e-02
            3.23649481112759998535644001549372e-02 3.23649481112759998535644001549372e-02 6.82986550133898515091424030742928e-03
            3.23649481112759998535644001549372e-02 9.35270103777448014170659007504582e-01 6.82986550133898515091424030742928e-03
            9.35270103777448014170659007504582e-01 3.23649481112759998535644001549372e-02 6.82986550133898515091424030742928e-03
            5.01781383104950018392287347523961e-02 3.56620648261293027303509006742388e-01 2.61685559811019423370748171464584e-02
            5.01781383104950018392287347523961e-02 5.93201213428211970857262258505216e-01 2.61685559811019423370748171464584e-02
            3.56620648261293027303509006742388e-01 5.01781383104950018392287347523961e-02 2.61685559811019423370748171464584e-02
            3.56620648261293027303509006742388e-01 5.93201213428211970857262258505216e-01 2.61685559811019423370748171464584e-02
            5.93201213428211970857262258505216e-01 5.01781383104950018392287347523961e-02 2.61685559811019423370748171464584e-02
            5.93201213428211970857262258505216e-01 3.56620648261293027303509006742388e-01 2.61685559811019423370748171464584e-02
            2.10220165361660014935463181018349e-02 1.71488980304041999103148441463418e-01 1.03538298195704776133441171737104e-02
            2.10220165361660014935463181018349e-02 8.07489003159791995933858288481133e-01 1.03538298195704776133441171737104e-02
            1.71488980304041999103148441463418e-01 2.10220165361660014935463181018349e-02 1.03538298195704776133441171737104e-02
            1.71488980304041999103148441463418e-01 8.07489003159791995933858288481133e-01 1.03538298195704776133441171737104e-02
            8.07489003159791995933858288481133e-01 2.10220165361660014935463181018349e-02 1.03538298195704776133441171737104e-02
            8.07489003159791995933858288481133e-01 1.71488980304041999103148441463418e-01 1.03538298195704776133441171737104e-02
            ];
        
    case  12
        
        % ALG. DEG.:   12
        % PTS CARD.:   33
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.5e-15
        
        xyw=[
            4.88217389773805021668806602974655e-01 4.88217389773805021668806602974655e-01 1.28655332202275516501055463436387e-02
            4.88217389773805021668806602974655e-01 2.35652204523899566623867940506898e-02 1.28655332202275516501055463436387e-02
            2.35652204523899566623867940506898e-02 4.88217389773805021668806602974655e-01 1.28655332202275516501055463436387e-02
            4.39724392294459975882858771001338e-01 4.39724392294459975882858771001338e-01 2.18462722690190881691396640462699e-02
            4.39724392294459975882858771001338e-01 1.20551215411080048234282457997324e-01 2.18462722690190881691396640462699e-02
            1.20551215411080048234282457997324e-01 4.39724392294459975882858771001338e-01 2.18462722690190881691396640462699e-02
            2.71210385012115973157875714605325e-01 2.71210385012115973157875714605325e-01 3.14291121089426278567557915266661e-02
            2.71210385012115973157875714605325e-01 4.57579229975768053684248570789350e-01 3.14291121089426278567557915266661e-02
            4.57579229975768053684248570789350e-01 2.71210385012115973157875714605325e-01 3.14291121089426278567557915266661e-02
            1.27576145541586000176081938661810e-01 1.27576145541586000176081938661810e-01 1.73980564653545693998282217762608e-02
            1.27576145541586000176081938661810e-01 7.44847708916827944136684891418554e-01 1.73980564653545693998282217762608e-02
            7.44847708916827944136684891418554e-01 1.27576145541586000176081938661810e-01 1.73980564653545693998282217762608e-02
            2.13173504532099995778526846379464e-02 2.13173504532099995778526846379464e-02 3.08313052577951227442576431769794e-03
            2.13173504532099995778526846379464e-02 9.57365299093580035538764150260249e-01 3.08313052577951227442576431769794e-03
            9.57365299093580035538764150260249e-01 2.13173504532099995778526846379464e-02 3.08313052577951227442576431769794e-03
            1.15343494534697998954442255126196e-01 2.75713269685514017481864357250743e-01 2.01857788831905811399725081400902e-02
            1.15343494534697998954442255126196e-01 6.08943235779787928052542156365234e-01 2.01857788831905811399725081400902e-02
            2.75713269685514017481864357250743e-01 1.15343494534697998954442255126196e-01 2.01857788831905811399725081400902e-02
            2.75713269685514017481864357250743e-01 6.08943235779787928052542156365234e-01 2.01857788831905811399725081400902e-02
            6.08943235779787928052542156365234e-01 1.15343494534697998954442255126196e-01 2.01857788831905811399725081400902e-02
            6.08943235779787928052542156365234e-01 2.75713269685514017481864357250743e-01 2.01857788831905811399725081400902e-02
            2.28383322222570003812158745404304e-02 2.81325580989939982856640199315734e-01 1.11783866011515454347291509407114e-02
            2.28383322222570003812158745404304e-02 6.95836086787803065334401253494434e-01 1.11783866011515454347291509407114e-02
            2.81325580989939982856640199315734e-01 2.28383322222570003812158745404304e-02 1.11783866011515454347291509407114e-02
            2.81325580989939982856640199315734e-01 6.95836086787803065334401253494434e-01 1.11783866011515454347291509407114e-02
            6.95836086787803065334401253494434e-01 2.28383322222570003812158745404304e-02 1.11783866011515454347291509407114e-02
            6.95836086787803065334401253494434e-01 2.81325580989939982856640199315734e-01 1.11783866011515454347291509407114e-02
            2.57340505483300008293667815451045e-02 1.16251915907596994892436725876905e-01 8.65811555432953461330924938010867e-03
            2.57340505483300008293667815451045e-02 8.58014033544073018155984300392447e-01 8.65811555432953461330924938010867e-03
            1.16251915907596994892436725876905e-01 2.57340505483300008293667815451045e-02 8.65811555432953461330924938010867e-03
            1.16251915907596994892436725876905e-01 8.58014033544073018155984300392447e-01 8.65811555432953461330924938010867e-03
            8.58014033544073018155984300392447e-01 2.57340505483300008293667815451045e-02 8.65811555432953461330924938010867e-03
            8.58014033544073018155984300392447e-01 1.16251915907596994892436725876905e-01 8.65811555432953461330924938010867e-03
            ];
        
    case  13
        
        % ALG. DEG.:   13
        % PTS CARD.:   37
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 9.1e-15
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 2.62604617004010242042877365520326e-02
            4.95048184939704982721053738714545e-01 4.95048184939704982721053738714545e-01 5.64007260466500501272513190542668e-03
            4.95048184939704982721053738714545e-01 9.90363012059003455789252257090993e-03 5.64007260466500501272513190542668e-03
            9.90363012059003455789252257090993e-03 4.95048184939704982721053738714545e-01 5.64007260466500501272513190542668e-03
            4.68716635109574009110389170018607e-01 4.68716635109574009110389170018607e-01 1.57117591812270147866303915407116e-02
            4.68716635109574009110389170018607e-01 6.25667297808519817792216599627864e-02 1.57117591812270147866303915407116e-02
            6.25667297808519817792216599627864e-02 4.68716635109574009110389170018607e-01 1.57117591812270147866303915407116e-02
            4.14521336801277007211297132016625e-01 4.14521336801277007211297132016625e-01 2.35362512520970214302540313155987e-02
            4.14521336801277007211297132016625e-01 1.70957326397445985577405735966749e-01 2.35362512520970214302540313155987e-02
            1.70957326397445985577405735966749e-01 4.14521336801277007211297132016625e-01 2.35362512520970214302540313155987e-02
            2.29399572042831012552710490126628e-01 2.29399572042831012552710490126628e-01 2.36817932681775214498465231827140e-02
            2.29399572042831012552710490126628e-01 5.41200855914337974894579019746743e-01 2.36817932681775214498465231827140e-02
            5.41200855914337974894579019746743e-01 2.29399572042831012552710490126628e-01 2.36817932681775214498465231827140e-02
            1.14424495196329994994854928336281e-01 1.14424495196329994994854928336281e-01 1.55837645228970138322477367864849e-02
            1.14424495196329994994854928336281e-01 7.71151009607339954499138912069611e-01 1.55837645228970138322477367864849e-02
            7.71151009607339954499138912069611e-01 1.14424495196329994994854928336281e-01 1.55837645228970138322477367864849e-02
            2.48113913634590010914848789980169e-02 2.48113913634590010914848789980169e-02 3.98788573253700331933124800798396e-03
            2.48113913634590010914848789980169e-02 9.50377217273081997817030242003966e-01 3.98788573253700331933124800798396e-03
            9.50377217273081997817030242003966e-01 2.48113913634590010914848789980169e-02 3.98788573253700331933124800798396e-03
            9.48538283795789943786402886871656e-02 2.68794997058760976038627177331364e-01 1.84242013643660176858141142020031e-02
            9.48538283795789943786402886871656e-02 6.36351174561660015704944726167014e-01 1.84242013643660176858141142020031e-02
            2.68794997058760976038627177331364e-01 9.48538283795789943786402886871656e-02 1.84242013643660176858141142020031e-02
            2.68794997058760976038627177331364e-01 6.36351174561660015704944726167014e-01 1.84242013643660176858141142020031e-02
            6.36351174561660015704944726167014e-01 9.48538283795789943786402886871656e-02 1.84242013643660176858141142020031e-02
            6.36351174561660015704944726167014e-01 2.68794997058760976038627177331364e-01 1.84242013643660176858141142020031e-02
            1.81007732788069988782986285968946e-02 2.91730066734287973861938780828496e-01 8.70073165191100755810893474517798e-03
            1.81007732788069988782986285968946e-02 6.90169159986905089709807725739665e-01 8.70073165191100755810893474517798e-03
            2.91730066734287973861938780828496e-01 1.81007732788069988782986285968946e-02 8.70073165191100755810893474517798e-03
            2.91730066734287973861938780828496e-01 6.90169159986905089709807725739665e-01 8.70073165191100755810893474517798e-03
            6.90169159986905089709807725739665e-01 1.81007732788069988782986285968946e-02 8.70073165191100755810893474517798e-03
            6.90169159986905089709807725739665e-01 2.91730066734287973861938780828496e-01 8.70073165191100755810893474517798e-03
            2.22330766740900000189018470564406e-02 1.26357385491668999799586003973673e-01 7.76089341952250662270973435852284e-03
            2.22330766740900000189018470564406e-02 8.51409537834241003650959100923501e-01 7.76089341952250662270973435852284e-03
            1.26357385491668999799586003973673e-01 2.22330766740900000189018470564406e-02 7.76089341952250662270973435852284e-03
            1.26357385491668999799586003973673e-01 8.51409537834241003650959100923501e-01 7.76089341952250662270973435852284e-03
            8.51409537834241003650959100923501e-01 2.22330766740900000189018470564406e-02 7.76089341952250662270973435852284e-03
            8.51409537834241003650959100923501e-01 1.26357385491668999799586003973673e-01 7.76089341952250662270973435852284e-03
            ];
        
    case  14
        
        % ALG. DEG.:   14
        % PTS CARD.:   42
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.7e-15
        
        xyw=[
            4.88963910362179010604677387163974e-01 4.88963910362179010604677387163974e-01 1.09417906847144082854716984343213e-02
            4.88963910362179010604677387163974e-01 2.20721792756419787906452256720513e-02 1.09417906847144082854716984343213e-02
            2.20721792756419787906452256720513e-02 4.88963910362179010604677387163974e-01 1.09417906847144082854716984343213e-02
            4.17644719340453995926054631127045e-01 4.17644719340453995926054631127045e-01 1.63941767720623618465047854897421e-02
            4.17644719340453995926054631127045e-01 1.64710561319092008147890737745911e-01 1.63941767720623618465047854897421e-02
            1.64710561319092008147890737745911e-01 4.17644719340453995926054631127045e-01 1.63941767720623618465047854897421e-02
            2.73477528308838979675954306003405e-01 2.73477528308838979675954306003405e-01 2.58870522536457821349831220913984e-02
            2.73477528308838979675954306003405e-01 4.53044943382322040648091387993190e-01 2.58870522536457821349831220913984e-02
            4.53044943382322040648091387993190e-01 2.73477528308838979675954306003405e-01 2.58870522536457821349831220913984e-02
            1.77205532412542998699223062430974e-01 1.77205532412542998699223062430974e-01 2.10812943684963241541829859215795e-02
            1.77205532412542998699223062430974e-01 6.45588935174914002601553875138052e-01 2.10812943684963241541829859215795e-02
            6.45588935174914002601553875138052e-01 1.77205532412542998699223062430974e-01 2.10812943684963241541829859215795e-02
            6.17998830908730034661147101360257e-02 6.17998830908730034661147101360257e-02 7.21684983488843963955750027139402e-03
            6.17998830908730034661147101360257e-02 8.76400233818253937556619348470122e-01 7.21684983488843963955750027139402e-03
            8.76400233818253937556619348470122e-01 6.17998830908730034661147101360257e-02 7.21684983488843963955750027139402e-03
            1.93909612487009989922892572167257e-02 1.93909612487009989922892572167257e-02 2.46170180119997935794251553431877e-03
            1.93909612487009989922892572167257e-02 9.61218077502598022832103197288234e-01 2.46170180119997935794251553431877e-03
            9.61218077502598022832103197288234e-01 1.93909612487009989922892572167257e-02 2.46170180119997935794251553431877e-03
            5.71247574036480021408834772955743e-02 1.72266687821356012655371614528121e-01 1.23328766062818957760516269672735e-02
            5.71247574036480021408834772955743e-02 7.70608554774996012959320523805218e-01 1.23328766062818957760516269672735e-02
            1.72266687821356012655371614528121e-01 5.71247574036480021408834772955743e-02 1.23328766062818957760516269672735e-02
            1.72266687821356012655371614528121e-01 7.70608554774996012959320523805218e-01 1.23328766062818957760516269672735e-02
            7.70608554774996012959320523805218e-01 5.71247574036480021408834772955743e-02 1.23328766062818957760516269672735e-02
            7.70608554774996012959320523805218e-01 1.72266687821356012655371614528121e-01 1.23328766062818957760516269672735e-02
            9.29162493569719999397804599539086e-02 3.36861459796345019679364440889913e-01 1.92857553935303384362587308942238e-02
            9.29162493569719999397804599539086e-02 5.70222290846682966503067291341722e-01 1.92857553935303384362587308942238e-02
            3.36861459796345019679364440889913e-01 9.29162493569719999397804599539086e-02 1.92857553935303384362587308942238e-02
            3.36861459796345019679364440889913e-01 5.70222290846682966503067291341722e-01 1.92857553935303384362587308942238e-02
            5.70222290846682966503067291341722e-01 9.29162493569719999397804599539086e-02 1.92857553935303384362587308942238e-02
            5.70222290846682966503067291341722e-01 3.36861459796345019679364440889913e-01 1.92857553935303384362587308942238e-02
            1.46469500556539993646465447341143e-02 2.98372882136258010810081486852141e-01 7.21815405676693930270282351102651e-03
            1.46469500556539993646465447341143e-02 6.86980167808088015846124108065851e-01 7.21815405676693930270282351102651e-03
            2.98372882136258010810081486852141e-01 1.46469500556539993646465447341143e-02 7.21815405676693930270282351102651e-03
            2.98372882136258010810081486852141e-01 6.86980167808088015846124108065851e-01 7.21815405676693930270282351102651e-03
            6.86980167808088015846124108065851e-01 1.46469500556539993646465447341143e-02 7.21815405676693930270282351102651e-03
            6.86980167808088015846124108065851e-01 2.98372882136258010810081486852141e-01 7.21815405676693930270282351102651e-03
            1.26833093287199993870473235801910e-03 1.18974497696957004500006860325811e-01 2.50511441925047864329179070352893e-03
            1.26833093287199993870473235801910e-03 8.79757171370170953927924983872799e-01 2.50511441925047864329179070352893e-03
            1.18974497696957004500006860325811e-01 1.26833093287199993870473235801910e-03 2.50511441925047864329179070352893e-03
            1.18974497696957004500006860325811e-01 8.79757171370170953927924983872799e-01 2.50511441925047864329179070352893e-03
            8.79757171370170953927924983872799e-01 1.26833093287199993870473235801910e-03 2.50511441925047864329179070352893e-03
            8.79757171370170953927924983872799e-01 1.18974497696957004500006860325811e-01 2.50511441925047864329179070352893e-03
            ];
        
    case  15
        
        % ALG. DEG.:   15
        % PTS CARD.:   48
        % NEG. W.  :    0
        % OUT PTS. :    9
        % M.E.INF. : 3.6e-14
        
        xyw=[
            5.06972916858242972359960276662605e-01 5.06972916858242972359960276662605e-01 9.58437821424495289303335798791750e-04
            5.06972916858242972359960276662605e-01 -1.39458337164859447199205533252098e-02 9.58437821424495289303335798791750e-04
            -1.39458337164859447199205533252098e-02 5.06972916858242972359960276662605e-01 9.58437821424495289303335798791750e-04
            4.31406354283022985818263350665802e-01 4.31406354283022985818263350665802e-01 2.21245136355723917565008207475330e-02
            4.31406354283022985818263350665802e-01 1.37187291433954028363473298668396e-01 2.21245136355723917565008207475330e-02
            1.37187291433954028363473298668396e-01 4.31406354283022985818263350665802e-01 2.21245136355723917565008207475330e-02
            2.77693644847143983334092354198219e-01 2.77693644847143983334092354198219e-01 2.55932743594258736707924839492989e-02
            2.77693644847143983334092354198219e-01 4.44612710305712033331815291603561e-01 2.55932743594258736707924839492989e-02
            4.44612710305712033331815291603561e-01 2.77693644847143983334092354198219e-01 2.55932743594258736707924839492989e-02
            1.26464891041253996162296857619367e-01 1.26464891041253996162296857619367e-01 1.18438679353439421659821562116122e-02
            1.26464891041253996162296857619367e-01 7.47070217917491952164255053503439e-01 1.18438679353439421659821562116122e-02
            7.47070217917491952164255053503439e-01 1.26464891041253996162296857619367e-01 1.18438679353439421659821562116122e-02
            7.08083859746860000417001401729067e-02 7.08083859746860000417001401729067e-02 6.64488784501046821084901949916457e-03
            7.08083859746860000417001401729067e-02 8.58383228050628055427750950912014e-01 6.64488784501046821084901949916457e-03
            8.58383228050628055427750950912014e-01 7.08083859746860000417001401729067e-02 6.64488784501046821084901949916457e-03
            1.89651702410729983139248133738874e-02 1.89651702410729983139248133738874e-02 2.37445830409598827165362777691371e-03
            1.89651702410729983139248133738874e-02 9.62069659517853947860999141994398e-01 2.37445830409598827165362777691371e-03
            9.62069659517853947860999141994398e-01 1.89651702410729983139248133738874e-02 2.37445830409598827165362777691371e-03
            1.33734161966620995753984857401520e-01 2.61311371140087012054920023729210e-01 1.92750362997964053146660035054083e-02
            1.33734161966620995753984857401520e-01 6.04954466893292019946670734498184e-01 1.92750362997964053146660035054083e-02
            2.61311371140087012054920023729210e-01 1.33734161966620995753984857401520e-01 1.92750362997964053146660035054083e-02
            2.61311371140087012054920023729210e-01 6.04954466893292019946670734498184e-01 1.92750362997964053146660035054083e-02
            6.04954466893292019946670734498184e-01 1.33734161966620995753984857401520e-01 1.92750362997964053146660035054083e-02
            6.04954466893292019946670734498184e-01 2.61311371140087012054920023729210e-01 1.92750362997964053146660035054083e-02
            3.63666773969169992986039119386987e-02 5.75586555512814013191302819905104e-01 1.36079071603119345496013536944702e-02
            3.63666773969169992986039119386987e-02 3.88046767090268973632305460341740e-01 1.36079071603119345496013536944702e-02
            5.75586555512814013191302819905104e-01 3.63666773969169992986039119386987e-02 1.36079071603119345496013536944702e-02
            5.75586555512814013191302819905104e-01 3.88046767090268973632305460341740e-01 1.36079071603119345496013536944702e-02
            3.88046767090268973632305460341740e-01 3.63666773969169992986039119386987e-02 1.36079071603119345496013536944702e-02
            3.88046767090268973632305460341740e-01 5.75586555512814013191302819905104e-01 1.36079071603119345496013536944702e-02
            -1.01748831265709993587620374455582e-02 2.85712220049915976982646270698751e-01 1.09103868339849459888346583369412e-03
            -1.01748831265709993587620374455582e-02 7.24462663076655077887266998004634e-01 1.09103868339849459888346583369412e-03
            2.85712220049915976982646270698751e-01 -1.01748831265709993587620374455582e-02 1.09103868339849459888346583369412e-03
            2.85712220049915976982646270698751e-01 7.24462663076655077887266998004634e-01 1.09103868339849459888346583369412e-03
            7.24462663076655077887266998004634e-01 -1.01748831265709993587620374455582e-02 1.09103868339849459888346583369412e-03
            7.24462663076655077887266998004634e-01 2.85712220049915976982646270698751e-01 1.09103868339849459888346583369412e-03
            3.68438698758780033837290091014438e-02 2.15599664072284008531354970727989e-01 1.07526599238654471668574785780947e-02
            3.68438698758780033837290091014438e-02 7.47556466051838008901597731892252e-01 1.07526599238654471668574785780947e-02
            2.15599664072284008531354970727989e-01 3.68438698758780033837290091014438e-02 1.07526599238654471668574785780947e-02
            2.15599664072284008531354970727989e-01 7.47556466051838008901597731892252e-01 1.07526599238654471668574785780947e-02
            7.47556466051838008901597731892252e-01 3.68438698758780033837290091014438e-02 1.07526599238654471668574785780947e-02
            7.47556466051838008901597731892252e-01 2.15599664072284008531354970727989e-01 1.07526599238654471668574785780947e-02
            1.24598093311989992909571967061311e-02 1.03575616576386003719889572494139e-01 3.83697131552448132685295689725535e-03
            1.24598093311989992909571967061311e-02 8.83964574092415023010005370451836e-01 3.83697131552448132685295689725535e-03
            1.03575616576386003719889572494139e-01 1.24598093311989992909571967061311e-02 3.83697131552448132685295689725535e-03
            1.03575616576386003719889572494139e-01 8.83964574092415023010005370451836e-01 3.83697131552448132685295689725535e-03
            8.83964574092415023010005370451836e-01 1.24598093311989992909571967061311e-02 3.83697131552448132685295689725535e-03
            8.83964574092415023010005370451836e-01 1.03575616576386003719889572494139e-01 3.83697131552448132685295689725535e-03
            ];
        
    case  16
        
        % ALG. DEG.:   16
        % PTS CARD.:   52
        % NEG. W.  :    0
        % OUT PTS. :    6
        % M.E.INF. : 2.0e-14
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 2.34378487138210515883240958601164e-02
            4.97380541948438004151711311351391e-01 4.97380541948438004151711311351391e-01 3.20293928929250674986217717332693e-03
            4.97380541948438004151711311351391e-01 5.23891610312399169657737729721703e-03 3.20293928929250674986217717332693e-03
            5.23891610312399169657737729721703e-03 4.97380541948438004151711311351391e-01 3.20293928929250674986217717332693e-03
            4.13469438549351975265011560622952e-01 4.13469438549351975265011560622952e-01 2.08551483696935435752095600037137e-02
            4.13469438549351975265011560622952e-01 1.73061122901296049469976878754096e-01 2.08551483696935435752095600037137e-02
            1.73061122901296049469976878754096e-01 4.13469438549351975265011560622952e-01 2.08551483696935435752095600037137e-02
            4.70458599066991001169668606962659e-01 4.70458599066991001169668606962659e-01 1.34457421250320299788150180120283e-02
            4.70458599066991001169668606962659e-01 5.90828018660179976606627860746812e-02 1.34457421250320299788150180120283e-02
            5.90828018660179976606627860746812e-02 4.70458599066991001169668606962659e-01 1.34457421250320299788150180120283e-02
            2.40553749969521002416428245851421e-01 2.40553749969521002416428245851421e-01 2.10662613808250444302760939763175e-02
            2.40553749969521002416428245851421e-01 5.18892500060957995167143508297158e-01 2.10662613808250444302760939763175e-02
            5.18892500060957995167143508297158e-01 2.40553749969521002416428245851421e-01 2.10662613808250444302760939763175e-02
            1.47965794222572999405684868179378e-01 1.47965794222572999405684868179378e-01 1.50001334213865321343117642527432e-02
            1.47965794222572999405684868179378e-01 7.04068411554854001188630263641244e-01 1.50001334213865321343117642527432e-02
            7.04068411554854001188630263641244e-01 1.47965794222572999405684868179378e-01 1.50001334213865321343117642527432e-02
            7.54651876574739949310455244813056e-02 7.54651876574739949310455244813056e-02 7.10004946251201543572229013534525e-03
            7.54651876574739949310455244813056e-02 8.49069624685051982382333335408475e-01 7.10004946251201543572229013534525e-03
            8.49069624685051982382333335408475e-01 7.54651876574739949310455244813056e-02 7.10004946251201543572229013534525e-03
            1.65964026230249993287824139542863e-02 1.65964026230249993287824139542863e-02 1.79123117563650393821439443797772e-03
            1.65964026230249993287824139542863e-02 9.66807194753950049914692499442026e-01 1.79123117563650393821439443797772e-03
            9.66807194753950049914692499442026e-01 1.65964026230249993287824139542863e-02 1.79123117563650393821439443797772e-03
            1.03575692245251996248533998823405e-01 2.96555596579887004171638409388834e-01 1.63865737303135333702286402512982e-02
            1.03575692245251996248533998823405e-01 5.99868711174861068968766630860046e-01 1.63865737303135333702286402512982e-02
            2.96555596579887004171638409388834e-01 1.03575692245251996248533998823405e-01 1.63865737303135333702286402512982e-02
            2.96555596579887004171638409388834e-01 5.99868711174861068968766630860046e-01 1.63865737303135333702286402512982e-02
            5.99868711174861068968766630860046e-01 1.03575692245251996248533998823405e-01 1.63865737303135333702286402512982e-02
            5.99868711174861068968766630860046e-01 2.96555596579887004171638409388834e-01 1.63865737303135333702286402512982e-02
            2.00834116554160016510088837549119e-02 3.37723063403079004629603332432453e-01 7.64915312422051723445504123333194e-03
            2.00834116554160016510088837549119e-02 6.42193524941504945147130456462037e-01 7.64915312422051723445504123333194e-03
            3.37723063403079004629603332432453e-01 2.00834116554160016510088837549119e-02 7.64915312422051723445504123333194e-03
            3.37723063403079004629603332432453e-01 6.42193524941504945147130456462037e-01 7.64915312422051723445504123333194e-03
            6.42193524941504945147130456462037e-01 2.00834116554160016510088837549119e-02 7.64915312422051723445504123333194e-03
            6.42193524941504945147130456462037e-01 3.37723063403079004629603332432453e-01 7.64915312422051723445504123333194e-03
            -4.34100261413900012641420289583039e-03 2.04748281642812002267817206302425e-01 1.19312209641950260977094622205641e-03
            -4.34100261413900012641420289583039e-03 7.99592720971327053369748227851233e-01 1.19312209641950260977094622205641e-03
            2.04748281642812002267817206302425e-01 -4.34100261413900012641420289583039e-03 1.19312209641950260977094622205641e-03
            2.04748281642812002267817206302425e-01 7.99592720971327053369748227851233e-01 1.19312209641950260977094622205641e-03
            7.99592720971327053369748227851233e-01 -4.34100261413900012641420289583039e-03 1.19312209641950260977094622205641e-03
            7.99592720971327053369748227851233e-01 2.04748281642812002267817206302425e-01 1.19312209641950260977094622205641e-03
            4.19417864680100005436713672679616e-02 1.89358492130623001781231096174452e-01 9.54239637794952050287022871088993e-03
            4.19417864680100005436713672679616e-02 7.68699721401366886652795074041933e-01 9.54239637794952050287022871088993e-03
            1.89358492130623001781231096174452e-01 4.19417864680100005436713672679616e-02 9.54239637794952050287022871088993e-03
            1.89358492130623001781231096174452e-01 7.68699721401366886652795074041933e-01 9.54239637794952050287022871088993e-03
            7.68699721401366886652795074041933e-01 4.19417864680100005436713672679616e-02 9.54239637794952050287022871088993e-03
            7.68699721401366886652795074041933e-01 1.89358492130623001781231096174452e-01 9.54239637794952050287022871088993e-03
            1.43173202306810000428161089303103e-02 8.52836156826569935507365016746917e-02 3.42502727327100801174575295249269e-03
            1.43173202306810000428161089303103e-02 9.00399064086662037631469956977526e-01 3.42502727327100801174575295249269e-03
            8.52836156826569935507365016746917e-02 1.43173202306810000428161089303103e-02 3.42502727327100801174575295249269e-03
            8.52836156826569935507365016746917e-02 9.00399064086662037631469956977526e-01 3.42502727327100801174575295249269e-03
            9.00399064086662037631469956977526e-01 1.43173202306810000428161089303103e-02 3.42502727327100801174575295249269e-03
            9.00399064086662037631469956977526e-01 8.52836156826569935507365016746917e-02 3.42502727327100801174575295249269e-03
            ];
        
    case  17
        
        % ALG. DEG.:   17
        % PTS CARD.:   61
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.1e-14
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 1.67185996454014863810133562083138e-02
            4.97170540556774009921525703248335e-01 4.97170540556774009921525703248335e-01 2.54670772025349795297799460058741e-03
            4.97170540556774009921525703248335e-01 5.65891888645198015694859350332990e-03 2.54670772025349795297799460058741e-03
            5.65891888645198015694859350332990e-03 4.97170540556774009921525703248335e-01 2.54670772025349795297799460058741e-03
            4.82176322624625008916865454011713e-01 4.82176322624625008916865454011713e-01 7.33543226381899293037092135705279e-03
            4.82176322624625008916865454011713e-01 3.56473547507499821662690919765737e-02 7.33543226381899293037092135705279e-03
            3.56473547507499821662690919765737e-02 4.82176322624625008916865454011713e-01 7.33543226381899293037092135705279e-03
            4.50239969020782004882619276031619e-01 4.50239969020782004882619276031619e-01 1.21754391768359900016793773147583e-02
            4.50239969020782004882619276031619e-01 9.95200619584359902347614479367621e-02 1.21754391768359900016793773147583e-02
            9.95200619584359902347614479367621e-02 4.50239969020782004882619276031619e-01 1.21754391768359900016793773147583e-02
            4.00266239377397026188987183559220e-01 4.00266239377397026188987183559220e-01 1.55537754344844854348384188824639e-02
            4.00266239377397026188987183559220e-01 1.99467521245205947622025632881559e-01 1.55537754344844854348384188824639e-02
            1.99467521245205947622025632881559e-01 4.00266239377397026188987183559220e-01 1.55537754344844854348384188824639e-02
            2.52141267970952986932076100856648e-01 2.52141267970952986932076100856648e-01 1.56285556093099867180917783571203e-02
            2.52141267970952986932076100856648e-01 4.95717464058094026135847798286704e-01 1.56285556093099867180917783571203e-02
            4.95717464058094026135847798286704e-01 2.52141267970952986932076100856648e-01 1.56285556093099867180917783571203e-02
            1.62047004658461002435387854347937e-01 1.62047004658461002435387854347937e-01 1.24078271698324898303367191942925e-02
            1.62047004658461002435387854347937e-01 6.75905990683077995129224291304126e-01 1.24078271698324898303367191942925e-02
            6.75905990683077995129224291304126e-01 1.62047004658461002435387854347937e-01 1.24078271698324898303367191942925e-02
            7.58758822607459998410561752280046e-02 7.58758822607459998410561752280046e-02 7.02803653527849393062298943846145e-03
            7.58758822607459998410561752280046e-02 8.48248235478508028073463265172904e-01 7.02803653527849393062298943846145e-03
            8.48248235478508028073463265172904e-01 7.58758822607459998410561752280046e-02 7.02803653527849393062298943846145e-03
            1.56547269678220003774615776137580e-02 1.56547269678220003774615776137580e-02 1.59733808688949843052062860238038e-03
            1.56547269678220003774615776137580e-02 9.68690546064356006183970748679712e-01 1.59733808688949843052062860238038e-03
            9.68690546064356006183970748679712e-01 1.56547269678220003774615776137580e-02 1.59733808688949843052062860238038e-03
            3.34319867363657985404046257826849e-01 6.55493203809422997352385209524073e-01 4.05982765949649670317311844769392e-03
            3.34319867363657985404046257826849e-01 1.01869288269189617324173013912514e-02 4.05982765949649670317311844769392e-03
            6.55493203809422997352385209524073e-01 3.34319867363657985404046257826849e-01 4.05982765949649670317311844769392e-03
            6.55493203809422997352385209524073e-01 1.01869288269189617324173013912514e-02 4.05982765949649670317311844769392e-03
            1.01869288269189617324173013912514e-02 3.34319867363657985404046257826849e-01 4.05982765949649670317311844769392e-03
            1.01869288269189617324173013912514e-02 6.55493203809422997352385209524073e-01 4.05982765949649670317311844769392e-03
            2.92221537796943986453612751574838e-01 5.72337590532020046296679538500030e-01 1.34028711415814880819619148155653e-02
            2.92221537796943986453612751574838e-01 1.35440871671036022760858941182960e-01 1.34028711415814880819619148155653e-02
            5.72337590532020046296679538500030e-01 2.92221537796943986453612751574838e-01 1.34028711415814880819619148155653e-02
            5.72337590532020046296679538500030e-01 1.35440871671036022760858941182960e-01 1.34028711415814880819619148155653e-02
            1.35440871671036022760858941182960e-01 2.92221537796943986453612751574838e-01 1.34028711415814880819619148155653e-02
            1.35440871671036022760858941182960e-01 5.72337590532020046296679538500030e-01 1.34028711415814880819619148155653e-02
            3.19574885423190013788996566290734e-01 6.26001190286228026415926706249593e-01 9.22999660541099116450869388472711e-03
            3.19574885423190013788996566290734e-01 5.44239242905819597950767274596728e-02 9.22999660541099116450869388472711e-03
            6.26001190286228026415926706249593e-01 3.19574885423190013788996566290734e-01 9.22999660541099116450869388472711e-03
            6.26001190286228026415926706249593e-01 5.44239242905819597950767274596728e-02 9.22999660541099116450869388472711e-03
            5.44239242905819597950767274596728e-02 3.19574885423190013788996566290734e-01 9.22999660541099116450869388472711e-03
            5.44239242905819597950767274596728e-02 6.26001190286228026415926706249593e-01 9.22999660541099116450869388472711e-03
            1.90704224192291993400516503243125e-01 7.96427214974070984965237585129216e-01 4.23843426716399677978142079837198e-03
            1.90704224192291993400516503243125e-01 1.28685608336369661230946803698316e-02 4.23843426716399677978142079837198e-03
            7.96427214974070984965237585129216e-01 1.90704224192291993400516503243125e-01 4.23843426716399677978142079837198e-03
            7.96427214974070984965237585129216e-01 1.28685608336369661230946803698316e-02 4.23843426716399677978142079837198e-03
            1.28685608336369661230946803698316e-02 1.90704224192291993400516503243125e-01 4.23843426716399677978142079837198e-03
            1.28685608336369661230946803698316e-02 7.96427214974070984965237585129216e-01 4.23843426716399677978142079837198e-03
            1.80483211648746005861809749148961e-01 7.52351005937728967154498604941182e-01 9.14639838501249066715637070501543e-03
            1.80483211648746005861809749148961e-01 6.71657824135249992281160302809440e-02 9.14639838501249066715637070501543e-03
            7.52351005937728967154498604941182e-01 1.80483211648746005861809749148961e-01 9.14639838501249066715637070501543e-03
            7.52351005937728967154498604941182e-01 6.71657824135249992281160302809440e-02 9.14639838501249066715637070501543e-03
            6.71657824135249992281160302809440e-02 1.80483211648746005861809749148961e-01 9.14639838501249066715637070501543e-03
            6.71657824135249992281160302809440e-02 7.52351005937728967154498604941182e-01 9.14639838501249066715637070501543e-03
            8.07113136795639951603931194767938e-02 9.04625504095608046206677954614861e-01 3.33281600208249709910157676517883e-03
            8.07113136795639951603931194767938e-02 1.46631822248279863885045415372588e-02 3.33281600208249709910157676517883e-03
            9.04625504095608046206677954614861e-01 8.07113136795639951603931194767938e-02 3.33281600208249709910157676517883e-03
            9.04625504095608046206677954614861e-01 1.46631822248279863885045415372588e-02 3.33281600208249709910157676517883e-03
            1.46631822248279863885045415372588e-02 8.07113136795639951603931194767938e-02 3.33281600208249709910157676517883e-03
            1.46631822248279863885045415372588e-02 9.04625504095608046206677954614861e-01 3.33281600208249709910157676517883e-03
            ];
        
    case  18
        
        % ALG. DEG.:   18
        % PTS CARD.:   70
        % NEG. W.  :    3
        % OUT PTS. :    6
        % M.E.INF. : 5.1e-13
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 1.54049699688234356664828794691857e-02
            4.93344808630921005665470602252753e-01 4.93344808630921005665470602252753e-01 4.53621833970198080204783153135395e-03
            4.93344808630921005665470602252753e-01 1.33103827381579886690587954944931e-02 4.53621833970198080204783153135395e-03
            1.33103827381579886690587954944931e-02 4.93344808630921005665470602252753e-01 4.53621833970198080204783153135395e-03
            4.69210594241956990835262786276871e-01 4.69210594241956990835262786276871e-01 9.38065846979696006668891783419895e-03
            4.69210594241956990835262786276871e-01 6.15788115160860183294744274462573e-02 9.38065846979696006668891783419895e-03
            6.15788115160860183294744274462573e-02 4.69210594241956990835262786276871e-01 9.38065846979696006668891783419895e-03
            4.36281395887005984945972159039229e-01 4.36281395887005984945972159039229e-01 9.72054899273845814111627561260320e-03
            4.36281395887005984945972159039229e-01 1.27437208225988030108055681921542e-01 9.72054899273845814111627561260320e-03
            1.27437208225988030108055681921542e-01 4.36281395887005984945972159039229e-01 9.72054899273845814111627561260320e-03
            3.94846170673415985952203755005030e-01 3.94846170673415985952203755005030e-01 1.38769743054049411734585106614759e-02
            3.94846170673415985952203755005030e-01 2.10307658653168028095592489989940e-01 1.38769743054049411734585106614759e-02
            2.10307658653168028095592489989940e-01 3.94846170673415985952203755005030e-01 1.38769743054049411734585106614759e-02
            2.49794568803157007730320060545637e-01 2.49794568803157007730320060545637e-01 1.61281126757284315464957558106107e-02
            2.49794568803157007730320060545637e-01 5.00410862393686040050511110166553e-01 1.61281126757284315464957558106107e-02
            5.00410862393686040050511110166553e-01 2.49794568803157007730320060545637e-01 1.61281126757284315464957558106107e-02
            1.61432193743842988808978589077014e-01 1.61432193743842988808978589077014e-01 1.25370163084609475417918744710732e-02
            1.61432193743842988808978589077014e-01 6.77135612512314022382042821845971e-01 1.25370163084609475417918744710732e-02
            6.77135612512314022382042821845971e-01 1.61432193743842988808978589077014e-01 1.25370163084609475417918744710732e-02
            7.65982274853709982354033058982168e-02 7.65982274853709982354033058982168e-02 7.63596398591596774163559047110539e-03
            7.65982274853709982354033058982168e-02 8.46803545029258031284769003832480e-01 7.63596398591596774163559047110539e-03
            8.46803545029258031284769003832480e-01 7.65982274853709982354033058982168e-02 7.63596398591596774163559047110539e-03
            2.42524393534499990709996097848489e-02 2.42524393534499990709996097848489e-02 3.39696101148148589185526979861152e-03
            2.42524393534499990709996097848489e-02 9.51495121293099987980212972615846e-01 3.39696101148148589185526979861152e-03
            9.51495121293099987980212972615846e-01 2.42524393534499990709996097848489e-02 3.39696101148148589185526979861152e-03
            4.31463672169649994914131241330324e-02 4.31463672169649994914131241330324e-02 -1.11154936495999525497557769426749e-03
            4.31463672169649994914131241330324e-02 9.13707265566069959383810328290565e-01 -1.11154936495999525497557769426749e-03
            9.13707265566069959383810328290565e-01 4.31463672169649994914131241330324e-02 -1.11154936495999525497557769426749e-03
            3.58911494940943986975412371975835e-01 6.32657968856636054866271479113493e-01 3.16595703820298640662755929042760e-03
            3.58911494940943986975412371975835e-01 8.43053620241995815831614891067147e-03 3.16595703820298640662755929042760e-03
            6.32657968856636054866271479113493e-01 3.58911494940943986975412371975835e-01 3.16595703820298640662755929042760e-03
            6.32657968856636054866271479113493e-01 8.43053620241995815831614891067147e-03 3.16595703820298640662755929042760e-03
            8.43053620241995815831614891067147e-03 3.58911494940943986975412371975835e-01 3.16595703820298640662755929042760e-03
            8.43053620241995815831614891067147e-03 6.32657968856636054866271479113493e-01 3.16595703820298640662755929042760e-03
            2.94402476751957020706385037556174e-01 5.74410971510854961863401513255667e-01 1.36287690245689430135200836957665e-02
            2.94402476751957020706385037556174e-01 1.31186551737187961919062217930332e-01 1.36287690245689430135200836957665e-02
            5.74410971510854961863401513255667e-01 2.94402476751957020706385037556174e-01 1.36287690245689430135200836957665e-02
            5.74410971510854961863401513255667e-01 1.31186551737187961919062217930332e-01 1.36287690245689430135200836957665e-02
            1.31186551737187961919062217930332e-01 2.94402476751957020706385037556174e-01 1.36287690245689430135200836957665e-02
            1.31186551737187961919062217930332e-01 5.74410971510854961863401513255667e-01 1.36287690245689430135200836957665e-02
            3.25017801641813985913387341497582e-01 6.24779046792511949526272019284079e-01 8.83839282473246334870253804183449e-03
            3.25017801641813985913387341497582e-01 5.02031515656741200714918704761658e-02 8.83839282473246334870253804183449e-03
            6.24779046792511949526272019284079e-01 3.25017801641813985913387341497582e-01 8.83839282473246334870253804183449e-03
            6.24779046792511949526272019284079e-01 5.02031515656741200714918704761658e-02 8.83839282473246334870253804183449e-03
            5.02031515656741200714918704761658e-02 3.25017801641813985913387341497582e-01 8.83839282473246334870253804183449e-03
            5.02031515656741200714918704761658e-02 6.24779046792511949526272019284079e-01 8.83839282473246334870253804183449e-03
            1.84737559666045991768612566374941e-01 7.48933176523037036353969142510323e-01 9.18974231903496238671458229418931e-03
            1.84737559666045991768612566374941e-01 6.63292638109169718774182911147363e-02 9.18974231903496238671458229418931e-03
            7.48933176523037036353969142510323e-01 1.84737559666045991768612566374941e-01 9.18974231903496238671458229418931e-03
            7.48933176523037036353969142510323e-01 6.63292638109169718774182911147363e-02 9.18974231903496238671458229418931e-03
            6.63292638109169718774182911147363e-02 1.84737559666045991768612566374941e-01 9.18974231903496238671458229418931e-03
            6.63292638109169718774182911147363e-02 7.48933176523037036353969142510323e-01 9.18974231903496238671458229418931e-03
            2.18796800013320996702859133620223e-01 7.69207005420442957444038256653585e-01 4.05236640409598232459575939401475e-03
            2.18796800013320996702859133620223e-01 1.19961945662360180975269940972794e-02 4.05236640409598232459575939401475e-03
            7.69207005420442957444038256653585e-01 2.18796800013320996702859133620223e-01 4.05236640409598232459575939401475e-03
            7.69207005420442957444038256653585e-01 1.19961945662360180975269940972794e-02 4.05236640409598232459575939401475e-03
            1.19961945662360180975269940972794e-02 2.18796800013320996702859133620223e-01 4.05236640409598232459575939401475e-03
            1.19961945662360180975269940972794e-02 7.69207005420442957444038256653585e-01 4.05236640409598232459575939401475e-03
            1.01179597136408006008423399180174e-01 8.83962302273466948854263591783820e-01 3.81706453536248416572029817928069e-03
            1.01179597136408006008423399180174e-01 1.48581005901250451373130090360064e-02 3.81706453536248416572029817928069e-03
            8.83962302273466948854263591783820e-01 1.01179597136408006008423399180174e-01 3.81706453536248416572029817928069e-03
            8.83962302273466948854263591783820e-01 1.48581005901250451373130090360064e-02 3.81706453536248416572029817928069e-03
            1.48581005901250451373130090360064e-02 1.01179597136408006008423399180174e-01 3.81706453536248416572029817928069e-03
            1.48581005901250451373130090360064e-02 8.83962302273466948854263591783820e-01 3.81706453536248416572029817928069e-03
            2.08747552825860015202064090544809e-02 1.01434726000536290513309722882695e+00 2.30938303969999016542532371820684e-05
            2.08747552825860015202064090544809e-02 -3.52220152879488823671749742061365e-02 2.30938303969999016542532371820684e-05
            1.01434726000536290513309722882695e+00 2.08747552825860015202064090544809e-02 2.30938303969999016542532371820684e-05
            1.01434726000536290513309722882695e+00 -3.52220152879488823671749742061365e-02 2.30938303969999016542532371820684e-05
            -3.52220152879488823671749742061365e-02 2.08747552825860015202064090544809e-02 2.30938303969999016542532371820684e-05
            -3.52220152879488823671749742061365e-02 1.01434726000536290513309722882695e+00 2.30938303969999016542532371820684e-05
            ];
        
    case  19
        
        % ALG. DEG.:   19
        % PTS CARD.:   73
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.6e-14
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 1.64531656944594158187289423267430e-02
            4.89609987073006014401954644199577e-01 4.89609987073006014401954644199577e-01 5.16536594563597392659737295161904e-03
            4.89609987073006014401954644199577e-01 2.07800258539879711960907116008457e-02 5.16536594563597392659737295161904e-03
            2.07800258539879711960907116008457e-02 4.89609987073006014401954644199577e-01 5.16536594563597392659737295161904e-03
            4.54536892697892991321850786334835e-01 4.54536892697892991321850786334835e-01 1.11936236315079429282093315123348e-02
            4.54536892697892991321850786334835e-01 9.09262146042140173562984273303300e-02 1.11936236315079429282093315123348e-02
            9.09262146042140173562984273303300e-02 4.54536892697892991321850786334835e-01 1.11936236315079429282093315123348e-02
            4.01416680649430979155312115835841e-01 4.01416680649430979155312115835841e-01 1.51330629347339214174628096998276e-02
            4.01416680649430979155312115835841e-01 1.97166638701138041689375768328318e-01 1.51330629347339214174628096998276e-02
            1.97166638701138041689375768328318e-01 4.01416680649430979155312115835841e-01 1.51330629347339214174628096998276e-02
            2.55551654403098016388184987590648e-01 2.55551654403098016388184987590648e-01 1.52454839010989218195391003973782e-02
            2.55551654403098016388184987590648e-01 4.88896691193803967223630024818704e-01 1.52454839010989218195391003973782e-02
            4.88896691193803967223630024818704e-01 2.55551654403098016388184987590648e-01 1.52454839010989218195391003973782e-02
            1.77077942152129991226772176560189e-01 1.77077942152129991226772176560189e-01 1.20796063708204382686206557195874e-02
            1.77077942152129991226772176560189e-01 6.45844115695739962035304415621795e-01 1.20796063708204382686206557195874e-02
            6.45844115695739962035304415621795e-01 1.77077942152129991226772176560189e-01 1.20796063708204382686206557195874e-02
            1.10061053227951996524858202519681e-01 1.10061053227951996524858202519681e-01 8.02540179340045789491142613769625e-03
            1.10061053227951996524858202519681e-01 7.79877893544095979194707979331724e-01 8.02540179340045789491142613769625e-03
            7.79877893544095979194707979331724e-01 1.10061053227951996524858202519681e-01 8.02540179340045789491142613769625e-03
            5.55286242518399988932209510039684e-02 5.55286242518399988932209510039684e-02 4.04229013089197914548877932361393e-03
            5.55286242518399988932209510039684e-02 8.88942751496319960580194674548693e-01 4.04229013089197914548877932361393e-03
            8.88942751496319960580194674548693e-01 5.55286242518399988932209510039684e-02 4.04229013089197914548877932361393e-03
            1.26218637772290004361863680060196e-02 1.26218637772290004361863680060196e-02 1.03968101374249475427058442278394e-03
            1.26218637772290004361863680060196e-02 9.74756272445541971372051648359047e-01 1.03968101374249475427058442278394e-03
            9.74756272445541971372051648359047e-01 1.26218637772290004361863680060196e-02 1.03968101374249475427058442278394e-03
            3.95754787356943027454292405309388e-01 6.00633794794644959580409704358317e-01 1.94243845249049006292108021654030e-03
            3.95754787356943027454292405309388e-01 3.61141784841201296529789033229463e-03 1.94243845249049006292108021654030e-03
            6.00633794794644959580409704358317e-01 3.95754787356943027454292405309388e-01 1.94243845249049006292108021654030e-03
            6.00633794794644959580409704358317e-01 3.61141784841201296529789033229463e-03 1.94243845249049006292108021654030e-03
            3.61141784841201296529789033229463e-03 3.95754787356943027454292405309388e-01 1.94243845249049006292108021654030e-03
            3.61141784841201296529789033229463e-03 6.00633794794644959580409704358317e-01 1.94243845249049006292108021654030e-03
            3.07929983880436020093895876925671e-01 5.57603261588784038238486573391128e-01 1.27870803060109346216410841634570e-02
            3.07929983880436020093895876925671e-01 1.34466754530779941667617549683200e-01 1.27870803060109346216410841634570e-02
            5.57603261588784038238486573391128e-01 3.07929983880436020093895876925671e-01 1.27870803060109346216410841634570e-02
            5.57603261588784038238486573391128e-01 1.34466754530779941667617549683200e-01 1.27870803060109346216410841634570e-02
            1.34466754530779941667617549683200e-01 3.07929983880436020093895876925671e-01 1.27870803060109346216410841634570e-02
            1.34466754530779941667617549683200e-01 5.57603261588784038238486573391128e-01 1.27870803060109346216410841634570e-02
            2.64566948406520008330033988386276e-01 7.20987025817364957092081567679998e-01 4.44045178666897712854710889018861e-03
            2.64566948406520008330033988386276e-01 1.44460257761149790667332126758993e-02 4.44045178666897712854710889018861e-03
            7.20987025817364957092081567679998e-01 2.64566948406520008330033988386276e-01 4.44045178666897712854710889018861e-03
            7.20987025817364957092081567679998e-01 1.44460257761149790667332126758993e-02 4.44045178666897712854710889018861e-03
            1.44460257761149790667332126758993e-02 2.64566948406520008330033988386276e-01 4.44045178666897712854710889018861e-03
            1.44460257761149790667332126758993e-02 7.20987025817364957092081567679998e-01 4.44045178666897712854710889018861e-03
            3.58539352205951011853812815388665e-01 5.94527068955871040856209219782613e-01 8.06227338086545897155588136229198e-03
            3.58539352205951011853812815388665e-01 4.69335788381779472899779648287222e-02 8.06227338086545897155588136229198e-03
            5.94527068955871040856209219782613e-01 3.58539352205951011853812815388665e-01 8.06227338086545897155588136229198e-03
            5.94527068955871040856209219782613e-01 4.69335788381779472899779648287222e-02 8.06227338086545897155588136229198e-03
            4.69335788381779472899779648287222e-02 3.58539352205951011853812815388665e-01 8.06227338086545897155588136229198e-03
            4.69335788381779472899779648287222e-02 5.94527068955871040856209219782613e-01 8.06227338086545897155588136229198e-03
            1.57807405968594993694864569988567e-01 8.39331473680839046380697254789993e-01 1.24597090874549367572887703659035e-03
            1.57807405968594993694864569988567e-01 2.86112035056595992443817522143945e-03 1.24597090874549367572887703659035e-03
            8.39331473680839046380697254789993e-01 1.57807405968594993694864569988567e-01 1.24597090874549367572887703659035e-03
            8.39331473680839046380697254789993e-01 2.86112035056595992443817522143945e-03 1.24597090874549367572887703659035e-03
            2.86112035056595992443817522143945e-03 1.57807405968594993694864569988567e-01 1.24597090874549367572887703659035e-03
            2.86112035056595992443817522143945e-03 8.39331473680839046380697254789993e-01 1.24597090874549367572887703659035e-03
            7.50505969759110019845493866341712e-02 7.01087978926173027716117758245673e-01 9.12142005947545395083686514681176e-03
            7.50505969759110019845493866341712e-02 2.23861424097915984177120662934612e-01 9.12142005947545395083686514681176e-03
            7.01087978926173027716117758245673e-01 7.50505969759110019845493866341712e-02 9.12142005947545395083686514681176e-03
            7.01087978926173027716117758245673e-01 2.23861424097915984177120662934612e-01 9.12142005947545395083686514681176e-03
            2.23861424097915984177120662934612e-01 7.50505969759110019845493866341712e-02 9.12142005947545395083686514681176e-03
            2.23861424097915984177120662934612e-01 7.01087978926173027716117758245673e-01 9.12142005947545395083686514681176e-03
            1.42421601113383006209645031958644e-01 8.22931324069857028646879371081013e-01 5.12928186809947367352791047778737e-03
            1.42421601113383006209645031958644e-01 3.46470748167599928990512125892565e-02 5.12928186809947367352791047778737e-03
            8.22931324069857028646879371081013e-01 1.42421601113383006209645031958644e-01 5.12928186809947367352791047778737e-03
            8.22931324069857028646879371081013e-01 3.46470748167599928990512125892565e-02 5.12928186809947367352791047778737e-03
            3.46470748167599928990512125892565e-02 1.42421601113383006209645031958644e-01 5.12928186809947367352791047778737e-03
            3.46470748167599928990512125892565e-02 8.22931324069857028646879371081013e-01 5.12928186809947367352791047778737e-03
            6.54946280829380028665198665294156e-02 9.24344252620784012641763638384873e-01 1.89996442765099022090269365037329e-03
            6.54946280829380028665198665294156e-02 1.01611192962779428583530716423411e-02 1.89996442765099022090269365037329e-03
            9.24344252620784012641763638384873e-01 6.54946280829380028665198665294156e-02 1.89996442765099022090269365037329e-03
            9.24344252620784012641763638384873e-01 1.01611192962779428583530716423411e-02 1.89996442765099022090269365037329e-03
            1.01611192962779428583530716423411e-02 6.54946280829380028665198665294156e-02 1.89996442765099022090269365037329e-03
            1.01611192962779428583530716423411e-02 9.24344252620784012641763638384873e-01 1.89996442765099022090269365037329e-03
            ];
        
    case  20
        
        % ALG. DEG.:   20
        % PTS CARD.:   79
        % NEG. W.  :    3
        % OUT PTS. :    9
        % M.E.INF. : 4.7e-14
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 1.65285277708119397732655642130339e-02
            5.00950464352200031115103229240049e-01 5.00950464352200031115103229240049e-01 4.33509592831498463422895683905267e-04
            5.00950464352200031115103229240049e-01 -1.90092870440006223020645848009735e-03 4.33509592831498463422895683905267e-04
            -1.90092870440006223020645848009735e-03 5.00950464352200031115103229240049e-01 4.33509592831498463422895683905267e-04
            4.88212957934729019360275970029761e-01 4.88212957934729019360275970029761e-01 5.83002635822397943105022122267656e-03
            4.88212957934729019360275970029761e-01 2.35740841305419612794480599404778e-02 5.83002635822397943105022122267656e-03
            2.35740841305419612794480599404778e-02 4.88212957934729019360275970029761e-01 5.83002635822397943105022122267656e-03
            4.55136681950283006337087954307208e-01 4.55136681950283006337087954307208e-01 1.14384681782104605241023875805695e-02
            4.55136681950283006337087954307208e-01 8.97266360994339873258240913855843e-02 1.14384681782104605241023875805695e-02
            8.97266360994339873258240913855843e-02 4.55136681950283006337087954307208e-01 1.14384681782104605241023875805695e-02
            4.01996259318288973183541656908346e-01 4.01996259318288973183541656908346e-01 1.52244913369689455207023343064066e-02
            4.01996259318288973183541656908346e-01 1.96007481363422053632916686183307e-01 1.52244913369689455207023343064066e-02
            1.96007481363422053632916686183307e-01 4.01996259318288973183541656908346e-01 1.52244913369689455207023343064066e-02
            2.55892909759420972282129014274688e-01 2.55892909759420972282129014274688e-01 1.53124458626774460207053962790269e-02
            2.55892909759420972282129014274688e-01 4.88214180481158055435741971450625e-01 1.53124458626774460207053962790269e-02
            4.88214180481158055435741971450625e-01 2.55892909759420972282129014274688e-01 1.53124458626774460207053962790269e-02
            1.76488255995106008144901466039300e-01 1.76488255995106008144901466039300e-01 1.21840288383999566235171840844487e-02
            1.76488255995106008144901466039300e-01 6.47023488009788039221348299179226e-01 1.21840288383999566235171840844487e-02
            6.47023488009788039221348299179226e-01 1.76488255995106008144901466039300e-01 1.21840288383999566235171840844487e-02
            1.04170855336758003129027372324344e-01 1.04170855336758003129027372324344e-01 7.99871601601197246278474040082074e-03
            1.04170855336758003129027372324344e-01 7.91658289326483965986369639722398e-01 7.99871601601197246278474040082074e-03
            7.91658289326483965986369639722398e-01 1.04170855336758003129027372324344e-01 7.99871601601197246278474040082074e-03
            5.30689638409300029620041527778085e-02 5.30689638409300029620041527778085e-02 3.84915090780098625988792093721713e-03
            5.30689638409300029620041527778085e-02 8.93862072318139966320416078815470e-01 3.84915090780098625988792093721713e-03
            8.93862072318139966320416078815470e-01 5.30689638409300029620041527778085e-02 3.84915090780098625988792093721713e-03
            4.16187151960289991592389924335293e-02 4.16187151960289991592389924335293e-02 -3.16030248743998836216673176124914e-04
            4.16187151960289991592389924335293e-02 9.16762569607942001681522015132941e-01 -3.16030248743998836216673176124914e-04
            9.16762569607942001681522015132941e-01 4.16187151960289991592389924335293e-02 -3.16030248743998836216673176124914e-04
            1.15819214068219999286268873106565e-02 1.15819214068219999286268873106565e-02 8.75567150596496808316604454347498e-04
            1.15819214068219999286268873106565e-02 9.76836157186355968917723657796159e-01 8.75567150596496808316604454347498e-04
            9.76836157186355968917723657796159e-01 1.15819214068219999286268873106565e-02 8.75567150596496808316604454347498e-04
            3.44855770229000990756418332239264e-01 6.06402646106160014838337701803539e-01 8.23291959478797061300348048007436e-03
            3.44855770229000990756418332239264e-01 4.87415836648390499163951972150244e-02 8.23291959478797061300348048007436e-03
            6.06402646106160014838337701803539e-01 3.44855770229000990756418332239264e-01 8.23291959478797061300348048007436e-03
            6.06402646106160014838337701803539e-01 4.87415836648390499163951972150244e-02 8.23291959478797061300348048007436e-03
            4.87415836648390499163951972150244e-02 3.44855770229000990756418332239264e-01 8.23291959478797061300348048007436e-03
            4.87415836648390499163951972150244e-02 6.06402646106160014838337701803539e-01 8.23291959478797061300348048007436e-03
            3.77843269594853981008242271855124e-01 6.15842614456540982104115755646490e-01 2.41951677024249134362321456137579e-03
            3.77843269594853981008242271855124e-01 6.31411594860509239879320375621319e-03 2.41951677024249134362321456137579e-03
            6.15842614456540982104115755646490e-01 3.77843269594853981008242271855124e-01 2.41951677024249134362321456137579e-03
            6.15842614456540982104115755646490e-01 6.31411594860509239879320375621319e-03 2.41951677024249134362321456137579e-03
            6.31411594860509239879320375621319e-03 3.77843269594853981008242271855124e-01 2.41951677024249134362321456137579e-03
            6.31411594860509239879320375621319e-03 6.15842614456540982104115755646490e-01 2.41951677024249134362321456137579e-03
            3.06635479062356997026483895751880e-01 5.59048000390295007910879121482139e-01 1.29024532673249550662752938023914e-02
            3.06635479062356997026483895751880e-01 1.34316520547347995062636982765980e-01 1.29024532673249550662752938023914e-02
            5.59048000390295007910879121482139e-01 3.06635479062356997026483895751880e-01 1.29024532673249550662752938023914e-02
            5.59048000390295007910879121482139e-01 1.34316520547347995062636982765980e-01 1.29024532673249550662752938023914e-02
            1.34316520547347995062636982765980e-01 3.06635479062356997026483895751880e-01 1.29024532673249550662752938023914e-02
            1.34316520547347995062636982765980e-01 5.59048000390295007910879121482139e-01 1.29024532673249550662752938023914e-02
            2.49419362774741998345362503641809e-01 7.36606743262866014987366725108586e-01 4.23554552722048543306199164248937e-03
            2.49419362774741998345362503641809e-01 1.39738939623920144228463868785184e-02 4.23554552722048543306199164248937e-03
            7.36606743262866014987366725108586e-01 2.49419362774741998345362503641809e-01 4.23554552722048543306199164248937e-03
            7.36606743262866014987366725108586e-01 1.39738939623920144228463868785184e-02 4.23554552722048543306199164248937e-03
            1.39738939623920144228463868785184e-02 2.49419362774741998345362503641809e-01 4.23554552722048543306199164248937e-03
            1.39738939623920144228463868785184e-02 7.36606743262866014987366725108586e-01 4.23554552722048543306199164248937e-03
            2.12775724802801990964695733055123e-01 7.11675142287434003840473906166153e-01 9.17745705313996636143247798145239e-03
            2.12775724802801990964695733055123e-01 7.55491329097640607059815920365509e-02 9.17745705313996636143247798145239e-03
            7.11675142287434003840473906166153e-01 2.12775724802801990964695733055123e-01 9.17745705313996636143247798145239e-03
            7.11675142287434003840473906166153e-01 7.55491329097640607059815920365509e-02 9.17745705313996636143247798145239e-03
            7.55491329097640607059815920365509e-02 2.12775724802801990964695733055123e-01 9.17745705313996636143247798145239e-03
            7.55491329097640607059815920365509e-02 7.11675142287434003840473906166153e-01 9.17745705313996636143247798145239e-03
            1.46965436053239001390480211739487e-01 8.61402717154986952152739831944928e-01 3.52202338953998739159295672251915e-04
            1.46965436053239001390480211739487e-01 -8.36815320822592578764442805550061e-03 3.52202338953998739159295672251915e-04
            8.61402717154986952152739831944928e-01 1.46965436053239001390480211739487e-01 3.52202338953998739159295672251915e-04
            8.61402717154986952152739831944928e-01 -8.36815320822592578764442805550061e-03 3.52202338953998739159295672251915e-04
            -8.36815320822592578764442805550061e-03 1.46965436053239001390480211739487e-01 3.52202338953998739159295672251915e-04
            -8.36815320822592578764442805550061e-03 8.61402717154986952152739831944928e-01 3.52202338953998739159295672251915e-04
            1.37726978828923013464802238559059e-01 8.35586957912363037515035557589727e-01 5.05634246373098169419213476771802e-03
            1.37726978828923013464802238559059e-01 2.66860632587139212645865882223006e-02 5.05634246373098169419213476771802e-03
            8.35586957912363037515035557589727e-01 1.37726978828923013464802238559059e-01 5.05634246373098169419213476771802e-03
            8.35586957912363037515035557589727e-01 2.66860632587139212645865882223006e-02 5.05634246373098169419213476771802e-03
            2.66860632587139212645865882223006e-02 1.37726978828923013464802238559059e-01 5.05634246373098169419213476771802e-03
            2.66860632587139212645865882223006e-02 8.35586957912363037515035557589727e-01 5.05634246373098169419213476771802e-03
            5.96961091490069983844790613147779e-02 9.29756171556852972770457199658267e-01 1.78695469297499368033221855966985e-03
            5.96961091490069983844790613147779e-02 1.05477192941400010894881233980414e-02 1.78695469297499368033221855966985e-03
            9.29756171556852972770457199658267e-01 5.96961091490069983844790613147779e-02 1.78695469297499368033221855966985e-03
            9.29756171556852972770457199658267e-01 1.05477192941400010894881233980414e-02 1.78695469297499368033221855966985e-03
            1.05477192941400010894881233980414e-02 5.96961091490069983844790613147779e-02 1.78695469297499368033221855966985e-03
            1.05477192941400010894881233980414e-02 9.29756171556852972770457199658267e-01 1.78695469297499368033221855966985e-03
            ];
        
        
    otherwise
        xyw=[];
end




pointset_stats=[
    1    1 1.92e-16    0    0
    2    3 3.68e-16    0    0
    3    4 2.22e-16    1    0
    4    6 2.24e-15    0    0
    5    7 1.79e-13    0    0
    6   12 2.75e-15    0    0
    7   13 4.16e-15    1    0
    8   16 3.87e-15    0    0
    9   19 3.43e-15    0    0
    10   25 1.10e-14    0    0
    11   27 1.20e-13    0    3
    12   33 4.54e-15    0    0
    13   37 9.07e-15    0    0
    14   42 6.71e-15    0    0
    15   48 3.60e-14    0    9
    16   52 1.98e-14    0    6
    17   61 1.10e-14    0    0
    18   70 5.08e-13    3    6
    19   73 1.65e-14    0    0
    20   79 4.74e-14    3    9
    ];



