


function [xyw_bar,pointset_stats]=set_lether_barycentric(deg)

%--------------------------------------------------------------------
% CUBATURE RULE ON THE SIMPLEX [0 0; 0 1; 1 0] WITH ADE EQUAL TO deg.
% THE CUBATURE RULE IS DESCRIBED IN BARYCENTRICAL FORM.
%--------------------------------------------------------------------
% INPUT:
% deg: algebraic degree of precision (ADE) of the rule.
%--------------------------------------------------------------------
% OUTPUT:
% xyw_bar: MATRIX OF NODES xyw(:,1:3) IN BARYCENTRICAL COOR. AND WEIGHTS xyw(:,4).
%      REMARK: The sum of the weights makes 1.
%
% pointset_stats:
%    statistics of the pointset stored in a matrix. Each rows contains
%    degree, cardinality of the full pointset, moment error w.r.t.
%    orthn. Dubiner basis, number of points with negative weights,
%    number of points outside the domain.
%--------------------------------------------------------------------
% SOURCE:
% F.G. Lether,"
% "Computation of double integrals over a triangle",
% JCAM, vol 2, no 3, (1976), pp. 219-224
%--------------------------------------------------------------------
% | DEG | CARD | W- | PT.O|   M.E.   |Q.|
%--------------------------------------------------------------------
% |   1 |   4 |   0 |   0 | 4.44e-16 |PI|
% |   2 |   4 |   0 |   0 | 4.44e-16 |PI|
% |   3 |   9 |   0 |   0 | 3.89e-16 |PI|
% |   4 |   9 |   0 |   0 | 3.89e-16 |PI|
% |   5 |  16 |   0 |   0 | 5.20e-16 |PI|
% |   6 |  16 |   0 |   0 | 5.20e-16 |PI|
% |   7 |  25 |   0 |   0 | 5.97e-16 |PI|
% |   8 |  25 |   0 |   0 | 5.97e-16 |PI|
% |   9 |  36 |   0 |   0 | 7.29e-16 |PI|
% |  10 |  36 |   0 |   0 | 7.29e-16 |PI|
% |  11 |  49 |   0 |   0 | 1.44e-15 |PI|
% |  12 |  49 |   0 |   0 | 1.44e-15 |PI|
% |  13 |  64 |   0 |   0 | 9.99e-16 |PI|
% |  14 |  64 |   0 |   0 | 9.99e-16 |PI|
% |  15 |  81 |   0 |   0 | 1.55e-15 |PI|
% |  16 |  81 |   0 |   0 | 1.55e-15 |PI|
% |  17 | 100 |   0 |   0 | 9.37e-16 |PI|
% |  18 | 100 |   0 |   0 | 9.37e-16 |PI|
% |  19 | 121 |   0 |   0 | 1.12e-15 |PI|
% |  20 | 121 |   0 |   0 | 1.12e-15 |PI|
% |  21 | 144 |   0 |   0 | 1.32e-15 |PI|
% |  22 | 144 |   0 |   0 | 1.32e-15 |PI|
% |  23 | 169 |   0 |   0 | 1.11e-15 |PI|
% |  24 | 169 |   0 |   0 | 1.11e-15 |PI|
% |  25 | 196 |   0 |   0 | 1.39e-15 |PI|
% |  26 | 196 |   0 |   0 | 1.39e-15 |PI|
% |  27 | 225 |   0 |   0 | 1.33e-15 |PI|
% |  28 | 225 |   0 |   0 | 1.33e-15 |PI|
% |  29 | 256 |   0 |   0 | 1.08e-15 |PI|
% |  30 | 256 |   0 |   0 | 1.66e-15 |PI|
% |  31 | 289 |   0 |   0 | 1.05e-15 |PI|
% |  32 | 289 |   0 |   0 | 1.05e-15 |PI|
% |  33 | 324 |   0 |   0 | 1.20e-15 |PI|
% |  34 | 324 |   0 |   0 | 1.22e-15 |PI|
% |  35 | 361 |   0 |   0 | 1.77e-15 |PI|
% |  36 | 361 |   0 |   0 | 1.77e-15 |PI|
% |  37 | 400 |   0 |   0 | 1.46e-15 |PI|
% |  38 | 400 |   0 |   0 | 1.46e-15 |PI|
% |  39 | 441 |   0 |   0 | 1.43e-15 |PI|
% |  40 | 441 |   0 |   0 | 1.43e-15 |PI|
% |  41 | 484 |   0 |   0 | 1.88e-15 |PI|
% |  42 | 484 |   0 |   0 | 1.88e-15 |PI|
% |  43 | 529 |   0 |   0 | 1.16e-15 |PI|
% |  44 | 529 |   0 |   0 | 1.16e-15 |PI|
% |  45 | 576 |   0 |   0 | 1.61e-15 |PI|
% |  46 | 576 |   0 |   0 | 1.61e-15 |PI|
% |  47 | 625 |   0 |   0 | 1.49e-15 |PI|
% |  48 | 625 |   0 |   0 | 1.49e-15 |PI|
% |  49 | 676 |   0 |   0 | 1.19e-15 |PI|
% |  50 | 676 |   0 |   0 | 1.19e-15 |PI|
%--------------------------------------------------------------------
% LEGEND:
%--------------------------------------------------------------------
% DEG  : algebraic degree of precision
% CARD : cardinality of the pointset
% W-   : negative weights
% PT.O : points outside the simplex [0 0; 1 0; 0 1].
% M.E. : maximum moment error in inf norm.
% Q.: quality of the rule.
%        PI: pos. weights, points in simplex
%        NI: some neg. weights, points in simplex
%        P0: pos. weights, some points out of the simplex
%        N0: some neg. weights, some points out of the simplex
%--------------------------------------------------------------------


switch deg
    
    case   1
        
        % ALG. DEG.:    1
        % PTS CARD.:    4
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.4e-16
        
        xyw_bar=[
            7.88675134594812865529434020572808e-01 1.66666666666666685170383743752609e-01 4.46581987385204493001822356745834e-02 1.05662432702593511724131758455769e-01
            2.11324865405187134470565979427192e-01 6.22008467928146235870201508078026e-01 1.66666666666666629659232512494782e-01 3.94337567297406210720112085255096e-01
            7.88675134594812865529434020572808e-01 4.46581987385204562390761395818117e-02 1.66666666666666685170383743752609e-01 1.05662432702593511724131758455769e-01
            2.11324865405187134470565979427192e-01 1.66666666666666685170383743752609e-01 6.22008467928146124847899045562372e-01 3.94337567297406210720112085255096e-01
            ];
        
    case   2
        
        % ALG. DEG.:    2
        % PTS CARD.:    4
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.4e-16
        
        xyw_bar=[
            7.88675134594812865529434020572808e-01 1.66666666666666685170383743752609e-01 4.46581987385204493001822356745834e-02 1.05662432702593511724131758455769e-01
            2.11324865405187134470565979427192e-01 6.22008467928146235870201508078026e-01 1.66666666666666629659232512494782e-01 3.94337567297406210720112085255096e-01
            7.88675134594812865529434020572808e-01 4.46581987385204562390761395818117e-02 1.66666666666666685170383743752609e-01 1.05662432702593511724131758455769e-01
            2.11324865405187134470565979427192e-01 1.66666666666666685170383743752609e-01 6.22008467928146124847899045562372e-01 3.94337567297406210720112085255096e-01
            ];
        
    case   3
        
        % ALG. DEG.:    3
        % PTS CARD.:    9
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.9e-16
        
        xyw_bar=[
            8.87298334620741702138957407441922e-01 1.00000000000000033306690738754696e-01 1.27016653792582645543518538033823e-02 1.73922323116139608545349659607382e-02
            5.00000000000000000000000000000000e-01 4.43649167310370851069478703720961e-01 5.63508326896291489305212962790392e-02 1.23456790123456797148016050869046e-01
            1.12701665379258297861042592558078e-01 7.87298334620741724343417899945052e-01 9.99999999999999777955395074968692e-02 1.36928755342707181297257079677365e-01
            8.87298334620741702138957407441922e-01 5.63508326896291766860969119079527e-02 5.63508326896291211749456806501257e-02 2.78275716985823123872378914711589e-02
            5.00000000000000000000000000000000e-01 2.50000000000000000000000000000000e-01 2.50000000000000000000000000000000e-01 1.97530864197530686698911495113862e-01
            1.12701665379258297861042592558078e-01 4.43649167310370851069478703720961e-01 4.43649167310370851069478703720961e-01 2.19086008548331295786582018081390e-01
            8.87298334620741702138957407441922e-01 1.27016653792583148613326571307880e-02 9.99999999999999777955395074968692e-02 1.73922323116139608545349659607382e-02
            5.00000000000000000000000000000000e-01 5.63508326896291489305212962790392e-02 4.43649167310370851069478703720961e-01 1.23456790123456797148016050869046e-01
            1.12701665379258297861042592558078e-01 9.99999999999999916733273153113259e-02 7.87298334620741724343417899945052e-01 1.36928755342707181297257079677365e-01
            ];
        
    case   4
        
        % ALG. DEG.:    4
        % PTS CARD.:    9
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.9e-16
        
        xyw_bar=[
            8.87298334620741702138957407441922e-01 1.00000000000000033306690738754696e-01 1.27016653792582645543518538033823e-02 1.73922323116139608545349659607382e-02
            5.00000000000000000000000000000000e-01 4.43649167310370851069478703720961e-01 5.63508326896291489305212962790392e-02 1.23456790123456797148016050869046e-01
            1.12701665379258297861042592558078e-01 7.87298334620741724343417899945052e-01 9.99999999999999777955395074968692e-02 1.36928755342707181297257079677365e-01
            8.87298334620741702138957407441922e-01 5.63508326896291766860969119079527e-02 5.63508326896291211749456806501257e-02 2.78275716985823123872378914711589e-02
            5.00000000000000000000000000000000e-01 2.50000000000000000000000000000000e-01 2.50000000000000000000000000000000e-01 1.97530864197530686698911495113862e-01
            1.12701665379258297861042592558078e-01 4.43649167310370851069478703720961e-01 4.43649167310370851069478703720961e-01 2.19086008548331295786582018081390e-01
            8.87298334620741702138957407441922e-01 1.27016653792583148613326571307880e-02 9.99999999999999777955395074968692e-02 1.73922323116139608545349659607382e-02
            5.00000000000000000000000000000000e-01 5.63508326896291489305212962790392e-02 4.43649167310370851069478703720961e-01 1.23456790123456797148016050869046e-01
            1.12701665379258297861042592558078e-01 9.99999999999999916733273153113259e-02 7.87298334620741724343417899945052e-01 1.36928755342707181297257079677365e-01
            ];
        
    case   5
        
        % ALG. DEG.:    5
        % PTS CARD.:   16
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.2e-16
        
        xyw_bar=[
            9.30568155797026230757751363853458e-01 6.46110632135476897763837200727721e-02 4.82078098942607946586491607376956e-03 4.20073048894970338396070985709230e-03
            6.69990521792428239677974488586187e-01 3.07096311531159016361414160201093e-01 2.29131666764127439606113512127195e-02 3.74316306300255460803860785290453e-02
            3.30009478207571815833176742671640e-01 6.23471844265867214396337203652365e-01 4.65186775265610252816372849338222e-02 7.59942952959004602853099186177133e-02
            6.94318442029737692422486361465417e-02 8.65957092583478527103579835966229e-01 6.46110632135477036541715278872289e-02 5.63007661538514128651833345884370e-02
            9.30568155797026230757751363853458e-01 4.65186775265609420149104380470817e-02 2.29131666764128272273381980994600e-02 7.87537121746693415114126679554829e-03
            6.69990521792428239677974488586187e-01 2.21103222500738022526789450239448e-01 1.08906255706833737795236061174364e-01 7.01754105058670085703909080621088e-02
            3.30009478207571815833176742671640e-01 4.48887299291690244906760653975653e-01 2.21103222500737994771213834610535e-01 1.42471240999480164290957873163279e-01
            6.94318442029737692422486361465417e-02 6.23471844265867214396337203652365e-01 3.07096311531159016361414160201093e-01 1.05550554708459157216005053214758e-01
            9.30568155797026230757751363853458e-01 2.29131666764127751856339187952472e-02 4.65186775265609975260616693049087e-02 7.87537121746693068169431484193410e-03
            6.69990521792428239677974488586187e-01 1.08906255706833821061962908061105e-01 2.21103222500737939260062603352708e-01 7.01754105058669808148152924331953e-02
            3.30009478207571815833176742671640e-01 2.21103222500738022526789450239448e-01 4.48887299291690244906760653975653e-01 1.42471240999480108779806641905452e-01
            6.94318442029737692422486361465417e-02 3.07096311531159016361414160201093e-01 6.23471844265867214396337203652365e-01 1.05550554708459101704853821956931e-01
            9.30568155797026230757751363853458e-01 4.82078098942601875054325688552126e-03 6.46110632135477452875349513305991e-02 4.20073048894970511868418583389939e-03
            6.69990521792428239677974488586187e-01 2.29131666764127925328686785633181e-02 3.07096311531158960850262928943266e-01 3.74316306300255599581738863435021e-02
            3.30009478207571815833176742671640e-01 4.65186775265609767093799575832236e-02 6.23471844265867214396337203652365e-01 7.59942952959005019186733420610835e-02
            6.94318442029737692422486361465417e-02 6.46110632135477452875349513305991e-02 8.65957092583478527103579835966229e-01 5.63007661538514406207589502173505e-02
            ];
        
    case   6
        
        % ALG. DEG.:    6
        % PTS CARD.:   16
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.2e-16
        
        xyw_bar=[
            9.30568155797026230757751363853458e-01 6.46110632135476897763837200727721e-02 4.82078098942607946586491607376956e-03 4.20073048894970338396070985709230e-03
            6.69990521792428239677974488586187e-01 3.07096311531159016361414160201093e-01 2.29131666764127439606113512127195e-02 3.74316306300255460803860785290453e-02
            3.30009478207571815833176742671640e-01 6.23471844265867214396337203652365e-01 4.65186775265610252816372849338222e-02 7.59942952959004602853099186177133e-02
            6.94318442029737692422486361465417e-02 8.65957092583478527103579835966229e-01 6.46110632135477036541715278872289e-02 5.63007661538514128651833345884370e-02
            9.30568155797026230757751363853458e-01 4.65186775265609420149104380470817e-02 2.29131666764128272273381980994600e-02 7.87537121746693415114126679554829e-03
            6.69990521792428239677974488586187e-01 2.21103222500738022526789450239448e-01 1.08906255706833737795236061174364e-01 7.01754105058670085703909080621088e-02
            3.30009478207571815833176742671640e-01 4.48887299291690244906760653975653e-01 2.21103222500737994771213834610535e-01 1.42471240999480164290957873163279e-01
            6.94318442029737692422486361465417e-02 6.23471844265867214396337203652365e-01 3.07096311531159016361414160201093e-01 1.05550554708459157216005053214758e-01
            9.30568155797026230757751363853458e-01 2.29131666764127751856339187952472e-02 4.65186775265609975260616693049087e-02 7.87537121746693068169431484193410e-03
            6.69990521792428239677974488586187e-01 1.08906255706833821061962908061105e-01 2.21103222500737939260062603352708e-01 7.01754105058669808148152924331953e-02
            3.30009478207571815833176742671640e-01 2.21103222500738022526789450239448e-01 4.48887299291690244906760653975653e-01 1.42471240999480108779806641905452e-01
            6.94318442029737692422486361465417e-02 3.07096311531159016361414160201093e-01 6.23471844265867214396337203652365e-01 1.05550554708459101704853821956931e-01
            9.30568155797026230757751363853458e-01 4.82078098942601875054325688552126e-03 6.46110632135477452875349513305991e-02 4.20073048894970511868418583389939e-03
            6.69990521792428239677974488586187e-01 2.29131666764127925328686785633181e-02 3.07096311531158960850262928943266e-01 3.74316306300255599581738863435021e-02
            3.30009478207571815833176742671640e-01 4.65186775265609767093799575832236e-02 6.23471844265867214396337203652365e-01 7.59942952959005019186733420610835e-02
            6.94318442029737692422486361465417e-02 6.46110632135477452875349513305991e-02 8.65957092583478527103579835966229e-01 5.63007661538514406207589502173505e-02
            ];
        
    case   7
        
        % ALG. DEG.:    7
        % PTS CARD.:   25
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.0e-16
        
        xyw_bar=[
            9.53089922969331926339009442017414e-01 4.47095217036448597247577652069594e-02 2.20055532702321393623279277562688e-03 1.31663331460145794499227989149404e-03
            7.69234655052841609368385888956254e-01 2.19940124839678508017470903723733e-01 1.08252201074798826141432073200122e-02 1.30843950585038618777966590300821e-02
            5.00000000000000000000000000000000e-01 4.76544961484665963169504721008707e-01 2.34550385153340368304952789912932e-02 3.36962680968802136272088887380960e-02
            2.30765344947158446142765342301573e-01 7.33149798129653307299236075778026e-01 3.60848569231881910468473506625742e-02 4.36156049414960847454558745539543e-02
            4.69100770306680736609905579825863e-02 9.08380401265687087430933388532139e-01 4.47095217036448389080760534852743e-02 2.67505411166127880784326720231547e-02
            9.53089922969331926339009442017414e-01 3.60848569231881841079534467553458e-02 1.08252201074798895530371112272405e-02 2.65980136763887577144860330236043e-03
            7.69234655052841609368385888956254e-01 1.77512700518577398733910399641900e-01 5.32526444285809918977037114018458e-02 2.64324861640542439766843330062329e-02
            5.00000000000000000000000000000000e-01 3.84617327526420804684192944478127e-01 1.15382672473579195315807055521873e-01 6.80716331376877009651238381593430e-02
            2.30765344947158446142765342301573e-01 5.91721954534264127367748642427614e-01 1.77512700518577370978334784012986e-01 8.81102159479413005183090490390896e-02
            4.69100770306680736609905579825863e-02 7.33149798129653418321538538293680e-01 2.19940124839678508017470903723733e-01 5.40401986323610292184405068383057e-02
            9.53089922969331926339009442017414e-01 2.34550385153340368304952789912932e-02 2.34550385153340368304952789912932e-02 3.16138906414138807360325600370743e-03
            7.69234655052841609368385888956254e-01 1.15382672473579195315807055521873e-01 1.15382672473579195315807055521873e-01 3.14171478042698446020430935732293e-02
            5.00000000000000000000000000000000e-01 2.50000000000000000000000000000000e-01 2.50000000000000000000000000000000e-01 8.09086419753087676687286489141115e-02
            2.30765344947158446142765342301573e-01 3.84617327526420749173041713220300e-01 3.84617327526420749173041713220300e-01 1.04726118471105647533825333539426e-01
            4.69100770306680736609905579825863e-02 4.76544961484665963169504721008707e-01 4.76544961484665963169504721008707e-01 6.42311471296190261703884516464313e-02
            9.53089922969331926339009442017414e-01 1.08252201074798930224840631808547e-02 3.60848569231881771690595428481174e-02 2.65980136763887837353381726757107e-03
            7.69234655052841609368385888956254e-01 5.32526444285810127143854231235309e-02 1.77512700518577370978334784012986e-01 2.64324861640542682628129966815322e-02
            5.00000000000000000000000000000000e-01 1.15382672473579223071382671150786e-01 3.84617327526420749173041713220300e-01 6.80716331376877703540628772316268e-02
            2.30765344947158446142765342301573e-01 1.77512700518577398733910399641900e-01 5.91721954534264127367748642427614e-01 8.81102159479413837850358959258301e-02
            4.69100770306680736609905579825863e-02 2.19940124839678563528622134981561e-01 7.33149798129653307299236075778026e-01 5.40401986323610777906978341889044e-02
            9.53089922969331926339009442017414e-01 2.20055532702321220150931679881978e-03 4.47095217036448597247577652069594e-02 1.31663331460145772815184539439315e-03
            7.69234655052841609368385888956254e-01 1.08252201074798895530371112272405e-02 2.19940124839678508017470903723733e-01 1.30843950585038584083497070764679e-02
            5.00000000000000000000000000000000e-01 2.34550385153340368304952789912932e-02 4.76544961484665963169504721008707e-01 3.36962680968802066883149848308676e-02
            2.30765344947158446142765342301573e-01 3.60848569231881771690595428481174e-02 7.33149798129653307299236075778026e-01 4.36156049414960778065619706467260e-02
            4.69100770306680736609905579825863e-02 4.47095217036448597247577652069594e-02 9.08380401265687087430933388532139e-01 2.67505411166127811395387681159264e-02
            ];
        
    case   8
        
        % ALG. DEG.:    8
        % PTS CARD.:   25
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.0e-16
        
        xyw_bar=[
            9.53089922969331926339009442017414e-01 4.47095217036448597247577652069594e-02 2.20055532702321393623279277562688e-03 1.31663331460145794499227989149404e-03
            7.69234655052841609368385888956254e-01 2.19940124839678508017470903723733e-01 1.08252201074798826141432073200122e-02 1.30843950585038618777966590300821e-02
            5.00000000000000000000000000000000e-01 4.76544961484665963169504721008707e-01 2.34550385153340368304952789912932e-02 3.36962680968802136272088887380960e-02
            2.30765344947158446142765342301573e-01 7.33149798129653307299236075778026e-01 3.60848569231881910468473506625742e-02 4.36156049414960847454558745539543e-02
            4.69100770306680736609905579825863e-02 9.08380401265687087430933388532139e-01 4.47095217036448389080760534852743e-02 2.67505411166127880784326720231547e-02
            9.53089922969331926339009442017414e-01 3.60848569231881841079534467553458e-02 1.08252201074798895530371112272405e-02 2.65980136763887577144860330236043e-03
            7.69234655052841609368385888956254e-01 1.77512700518577398733910399641900e-01 5.32526444285809918977037114018458e-02 2.64324861640542439766843330062329e-02
            5.00000000000000000000000000000000e-01 3.84617327526420804684192944478127e-01 1.15382672473579195315807055521873e-01 6.80716331376877009651238381593430e-02
            2.30765344947158446142765342301573e-01 5.91721954534264127367748642427614e-01 1.77512700518577370978334784012986e-01 8.81102159479413005183090490390896e-02
            4.69100770306680736609905579825863e-02 7.33149798129653418321538538293680e-01 2.19940124839678508017470903723733e-01 5.40401986323610292184405068383057e-02
            9.53089922969331926339009442017414e-01 2.34550385153340368304952789912932e-02 2.34550385153340368304952789912932e-02 3.16138906414138807360325600370743e-03
            7.69234655052841609368385888956254e-01 1.15382672473579195315807055521873e-01 1.15382672473579195315807055521873e-01 3.14171478042698446020430935732293e-02
            5.00000000000000000000000000000000e-01 2.50000000000000000000000000000000e-01 2.50000000000000000000000000000000e-01 8.09086419753087676687286489141115e-02
            2.30765344947158446142765342301573e-01 3.84617327526420749173041713220300e-01 3.84617327526420749173041713220300e-01 1.04726118471105647533825333539426e-01
            4.69100770306680736609905579825863e-02 4.76544961484665963169504721008707e-01 4.76544961484665963169504721008707e-01 6.42311471296190261703884516464313e-02
            9.53089922969331926339009442017414e-01 1.08252201074798930224840631808547e-02 3.60848569231881771690595428481174e-02 2.65980136763887837353381726757107e-03
            7.69234655052841609368385888956254e-01 5.32526444285810127143854231235309e-02 1.77512700518577370978334784012986e-01 2.64324861640542682628129966815322e-02
            5.00000000000000000000000000000000e-01 1.15382672473579223071382671150786e-01 3.84617327526420749173041713220300e-01 6.80716331376877703540628772316268e-02
            2.30765344947158446142765342301573e-01 1.77512700518577398733910399641900e-01 5.91721954534264127367748642427614e-01 8.81102159479413837850358959258301e-02
            4.69100770306680736609905579825863e-02 2.19940124839678563528622134981561e-01 7.33149798129653307299236075778026e-01 5.40401986323610777906978341889044e-02
            9.53089922969331926339009442017414e-01 2.20055532702321220150931679881978e-03 4.47095217036448597247577652069594e-02 1.31663331460145772815184539439315e-03
            7.69234655052841609368385888956254e-01 1.08252201074798895530371112272405e-02 2.19940124839678508017470903723733e-01 1.30843950585038584083497070764679e-02
            5.00000000000000000000000000000000e-01 2.34550385153340368304952789912932e-02 4.76544961484665963169504721008707e-01 3.36962680968802066883149848308676e-02
            2.30765344947158446142765342301573e-01 3.60848569231881771690595428481174e-02 7.33149798129653307299236075778026e-01 4.36156049414960778065619706467260e-02
            4.69100770306680736609905579825863e-02 4.47095217036448597247577652069594e-02 9.08380401265687087430933388532139e-01 2.67505411166127811395387681159264e-02
            ];
        
    case   9
        
        % ALG. DEG.:    9
        % PTS CARD.:   36
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 7.3e-16
        
        xyw_bar=[
            9.66234757101575914006730272376444e-01 3.26251512704345617010837088400876e-02 1.14009162798952429218601878346817e-03 4.95540083901405289078245797185218e-04
            8.30604693233132240770544285624055e-01 1.63675633088031402539996861378313e-01 5.71967367883635668945885299763177e-03 5.23493271062233037765887289083366e-03
            6.19309593041598382967549696331844e-01 3.67836302898351286128075798842474e-01 1.28541040600503309043745048256824e-02 1.52590455677919026311606387480424e-02
            3.80690406958401561521299072410329e-01 5.98398454203224683389805704791797e-01 2.09111388383736995777439915400464e-02 2.48235130911114189322663747816478e-02
            1.69395306766867703718304483118118e-01 8.02559124013544566977884642255958e-01 2.80455692195877848149621058837511e-02 2.56687139755692773912532089752858e-02
            3.37652428984240304821184963657288e-02 9.33609605831141387000116083072498e-01 3.26251512704345270066141893039457e-02 1.41805007605888093447088493803676e-02
            9.66234757101575914006730272376444e-01 2.80455692195877813455151539301369e-02 5.71967367883630464775457369341893e-03 1.04346913680634142268521813434745e-03
            8.30604693233132240770544285624055e-01 1.40700536812226523197821848043532e-01 2.86947699546412360316338663324132e-02 1.10233074866234303373913050450028e-02
            6.19309593041598382967549696331844e-01 3.16203238688479437712430808460340e-01 6.44871682699221793200194952078164e-02 3.21312919466680765867572233673854e-02
            3.80690406958401561521299072410329e-01 5.14401454544652803058113477163715e-01 1.04908138496945579909436219168128e-01 5.22713915971258352710648864558607e-02
            1.69395306766867703718304483118118e-01 6.89904156420905856350600515725091e-01 1.40700536812226495442246232414618e-01 5.40511488074601545439357153099991e-02
            3.37652428984240304821184963657288e-02 8.02559124013544455955582179740304e-01 1.63675633088031458051148092636140e-01 2.98601775493853444087832826880913e-02
            9.66234757101575914006730272376444e-01 2.09111388383737759055769345195586e-02 1.28541040600503100876927931039972e-02 1.35339732910820390379691424698194e-03
            8.30604693233132240770544285624055e-01 1.04908138496945593787224026982585e-01 6.44871682699221654422316873933596e-02 1.42974184708478235100725939332733e-02
            6.19309593041598382967549696331844e-01 2.35765221008248176959298803012643e-01 1.44925185950153440073151500655513e-01 4.16748355725323274545424112602632e-02
            3.80690406958401561521299072410329e-01 3.83544372033350178252675277690287e-01 2.35765221008248204714874418641557e-01 6.77968895111160096966784749383805e-02
            1.69395306766867703718304483118118e-01 5.14401454544652914080415939679369e-01 3.16203238688479437712430808460340e-01 7.01052650729459408962540578613698e-02
            3.37652428984240304821184963657288e-02 5.98398454203224683389805704791797e-01 3.67836302898351230616924567584647e-01 3.87291613297951475836100598826306e-02
            9.66234757101575914006730272376444e-01 1.28541040600503638641205483850172e-02 2.09111388383737203944257032617315e-02 1.35339732910820498799908673248638e-03
            8.30604693233132240770544285624055e-01 6.44871682699221654422316873933596e-02 1.04908138496945593787224026982585e-01 1.42974184708478339184134497941159e-02
            6.19309593041598382967549696331844e-01 1.44925185950153412317575885026599e-01 2.35765221008248204714874418641557e-01 4.16748355725323621490119307964051e-02
            3.80690406958401561521299072410329e-01 2.35765221008248149203723187383730e-01 3.83544372033350233763826508948114e-01 6.77968895111160652078297061962076e-02
            1.69395306766867703718304483118118e-01 3.16203238688479437712430808460340e-01 5.14401454544652914080415939679369e-01 7.01052650729459964074052891191968e-02
            3.37652428984240304821184963657288e-02 3.67836302898351230616924567584647e-01 5.98398454203224683389805704791797e-01 3.87291613297951753391856755115441e-02
            9.66234757101575914006730272376444e-01 5.71967367883635842418232897443886e-03 2.80455692195877293038108746259240e-02 1.04346913680633968796174215754036e-03
            8.30604693233132240770544285624055e-01 2.86947699546412256232930104715706e-02 1.40700536812226523197821848043532e-01 1.10233074866234129901565452769319e-02
            6.19309593041598382967549696331844e-01 6.44871682699221515644438795789029e-02 3.16203238688479437712430808460340e-01 3.21312919466680280144998960167868e-02
            3.80690406958401561521299072410329e-01 1.04908138496945552153860603539215e-01 5.14401454544652803058113477163715e-01 5.22713915971257589432319434763485e-02
            1.69395306766867703718304483118118e-01 1.40700536812226495442246232414618e-01 6.89904156420905856350600515725091e-01 5.40511488074600712772088684232585e-02
            3.37652428984240304821184963657288e-02 1.63675633088031347028845630120486e-01 8.02559124013544566977884642255958e-01 2.98601775493852993059729072911068e-02
            9.66234757101575914006730272376444e-01 1.14009162798957806861377406448810e-03 3.26251512704345061899324775822606e-02 4.95540083901405831179332039937435e-04
            8.30604693233132240770544285624055e-01 5.71967367883634194430930719477146e-03 1.63675633088031430295572477007227e-01 5.23493271062233644919103880965849e-03
            6.19309593041598382967549696331844e-01 1.28541040600503239654806009184540e-02 3.67836302898351286128075798842474e-01 1.52590455677919199783953985161133e-02
            3.80690406958401561521299072410329e-01 2.09111388383737065166378954472748e-02 5.98398454203224683389805704791797e-01 2.48235130911114501572889423641755e-02
            1.69395306766867703718304483118118e-01 2.80455692195876911398944031361680e-02 8.02559124013544678000187104771612e-01 2.56687139755693086162757765578135e-02
            3.37652428984240304821184963657288e-02 3.26251512704344576176751502316620e-02 9.33609605831141498022418545588152e-01 1.41805007605888249572201331716315e-02
            ];
        
    case  10
        
        % ALG. DEG.:   10
        % PTS CARD.:   36
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 7.3e-16
        
        xyw_bar=[
            9.66234757101575914006730272376444e-01 3.26251512704345617010837088400876e-02 1.14009162798952429218601878346817e-03 4.95540083901405289078245797185218e-04
            8.30604693233132240770544285624055e-01 1.63675633088031402539996861378313e-01 5.71967367883635668945885299763177e-03 5.23493271062233037765887289083366e-03
            6.19309593041598382967549696331844e-01 3.67836302898351286128075798842474e-01 1.28541040600503309043745048256824e-02 1.52590455677919026311606387480424e-02
            3.80690406958401561521299072410329e-01 5.98398454203224683389805704791797e-01 2.09111388383736995777439915400464e-02 2.48235130911114189322663747816478e-02
            1.69395306766867703718304483118118e-01 8.02559124013544566977884642255958e-01 2.80455692195877848149621058837511e-02 2.56687139755692773912532089752858e-02
            3.37652428984240304821184963657288e-02 9.33609605831141387000116083072498e-01 3.26251512704345270066141893039457e-02 1.41805007605888093447088493803676e-02
            9.66234757101575914006730272376444e-01 2.80455692195877813455151539301369e-02 5.71967367883630464775457369341893e-03 1.04346913680634142268521813434745e-03
            8.30604693233132240770544285624055e-01 1.40700536812226523197821848043532e-01 2.86947699546412360316338663324132e-02 1.10233074866234303373913050450028e-02
            6.19309593041598382967549696331844e-01 3.16203238688479437712430808460340e-01 6.44871682699221793200194952078164e-02 3.21312919466680765867572233673854e-02
            3.80690406958401561521299072410329e-01 5.14401454544652803058113477163715e-01 1.04908138496945579909436219168128e-01 5.22713915971258352710648864558607e-02
            1.69395306766867703718304483118118e-01 6.89904156420905856350600515725091e-01 1.40700536812226495442246232414618e-01 5.40511488074601545439357153099991e-02
            3.37652428984240304821184963657288e-02 8.02559124013544455955582179740304e-01 1.63675633088031458051148092636140e-01 2.98601775493853444087832826880913e-02
            9.66234757101575914006730272376444e-01 2.09111388383737759055769345195586e-02 1.28541040600503100876927931039972e-02 1.35339732910820390379691424698194e-03
            8.30604693233132240770544285624055e-01 1.04908138496945593787224026982585e-01 6.44871682699221654422316873933596e-02 1.42974184708478235100725939332733e-02
            6.19309593041598382967549696331844e-01 2.35765221008248176959298803012643e-01 1.44925185950153440073151500655513e-01 4.16748355725323274545424112602632e-02
            3.80690406958401561521299072410329e-01 3.83544372033350178252675277690287e-01 2.35765221008248204714874418641557e-01 6.77968895111160096966784749383805e-02
            1.69395306766867703718304483118118e-01 5.14401454544652914080415939679369e-01 3.16203238688479437712430808460340e-01 7.01052650729459408962540578613698e-02
            3.37652428984240304821184963657288e-02 5.98398454203224683389805704791797e-01 3.67836302898351230616924567584647e-01 3.87291613297951475836100598826306e-02
            9.66234757101575914006730272376444e-01 1.28541040600503638641205483850172e-02 2.09111388383737203944257032617315e-02 1.35339732910820498799908673248638e-03
            8.30604693233132240770544285624055e-01 6.44871682699221654422316873933596e-02 1.04908138496945593787224026982585e-01 1.42974184708478339184134497941159e-02
            6.19309593041598382967549696331844e-01 1.44925185950153412317575885026599e-01 2.35765221008248204714874418641557e-01 4.16748355725323621490119307964051e-02
            3.80690406958401561521299072410329e-01 2.35765221008248149203723187383730e-01 3.83544372033350233763826508948114e-01 6.77968895111160652078297061962076e-02
            1.69395306766867703718304483118118e-01 3.16203238688479437712430808460340e-01 5.14401454544652914080415939679369e-01 7.01052650729459964074052891191968e-02
            3.37652428984240304821184963657288e-02 3.67836302898351230616924567584647e-01 5.98398454203224683389805704791797e-01 3.87291613297951753391856755115441e-02
            9.66234757101575914006730272376444e-01 5.71967367883635842418232897443886e-03 2.80455692195877293038108746259240e-02 1.04346913680633968796174215754036e-03
            8.30604693233132240770544285624055e-01 2.86947699546412256232930104715706e-02 1.40700536812226523197821848043532e-01 1.10233074866234129901565452769319e-02
            6.19309593041598382967549696331844e-01 6.44871682699221515644438795789029e-02 3.16203238688479437712430808460340e-01 3.21312919466680280144998960167868e-02
            3.80690406958401561521299072410329e-01 1.04908138496945552153860603539215e-01 5.14401454544652803058113477163715e-01 5.22713915971257589432319434763485e-02
            1.69395306766867703718304483118118e-01 1.40700536812226495442246232414618e-01 6.89904156420905856350600515725091e-01 5.40511488074600712772088684232585e-02
            3.37652428984240304821184963657288e-02 1.63675633088031347028845630120486e-01 8.02559124013544566977884642255958e-01 2.98601775493852993059729072911068e-02
            9.66234757101575914006730272376444e-01 1.14009162798957806861377406448810e-03 3.26251512704345061899324775822606e-02 4.95540083901405831179332039937435e-04
            8.30604693233132240770544285624055e-01 5.71967367883634194430930719477146e-03 1.63675633088031430295572477007227e-01 5.23493271062233644919103880965849e-03
            6.19309593041598382967549696331844e-01 1.28541040600503239654806009184540e-02 3.67836302898351286128075798842474e-01 1.52590455677919199783953985161133e-02
            3.80690406958401561521299072410329e-01 2.09111388383737065166378954472748e-02 5.98398454203224683389805704791797e-01 2.48235130911114501572889423641755e-02
            1.69395306766867703718304483118118e-01 2.80455692195876911398944031361680e-02 8.02559124013544678000187104771612e-01 2.56687139755693086162757765578135e-02
            3.37652428984240304821184963657288e-02 3.26251512704344576176751502316620e-02 9.33609605831141498022418545588152e-01 1.41805007605888249572201331716315e-02
            ];
        
    case  11
        
        % ALG. DEG.:   11
        % PTS CARD.:   49
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.4e-15
        
        xyw_bar=[
            9.74553956171379409667565596464556e-01 2.47985426820925627944269820090994e-02 6.47501146528027538007421526344842e-04 2.13318720711471262628175504794115e-04
            8.70765592799697119019697311159689e-01 1.25945902810518123304817095231556e-01 3.28850438978475767548559360875515e-03 2.34028282161511443776324981058679e-03
            7.02922575688698647589092161069857e-01 2.89517979151782289815031390389777e-01 7.55944515951906259587644854036625e-03 7.34393977644001282739649738573462e-03
            4.99999999999999944488848768742173e-01 4.87276978085689815856085260747932e-01 1.27230219143101841439147392520681e-02 1.35298576894818649990481773670581e-02
            2.97077424311301407922059070187970e-01 6.85035977019597064341382974816952e-01 1.78865986691014722254067237372510e-02 1.73766858095164809794308524715234e-02
            1.29234407200302769958000226324657e-01 8.48608053360861314118324116861913e-01 2.21575394388359159236756568134297e-02 1.57685387555045360141736665582357e-02
            2.54460438286207013547368660510983e-02 9.49755413489286826056456902733771e-01 2.47985426820924725888062312151305e-02 8.16985951116516034842263138671115e-03
            9.74553956171379409667565596464556e-01 2.21575394388359298014634646278864e-02 3.28850438978466053097093890755787e-03 4.60797867536058493483086540010163e-04
            8.70765592799697119019697311159689e-01 1.12532875195889128794313194248389e-01 1.67015320044137521859894945919223e-02 5.05533377490165886092299274423567e-03
            7.02922575688698647589092161069857e-01 2.58684799487837457920846873093979e-01 3.83926248234638944900609658361645e-02 1.58639231334693486807907447655452e-02
            4.99999999999999944488848768742173e-01 4.35382796399848670532151118095499e-01 6.46172036001513294678488819045015e-02 2.92263592739815461496455384349247e-02
            2.97077424311301407922059070187970e-01 6.12080793311859494565396744292229e-01 9.08417823768390420013929542619735e-02 3.75360387459828664935557185344805e-02
            1.29234407200302769958000226324657e-01 7.58232717603808059614323155983584e-01 1.12532875195889170427676617691759e-01 3.40622192391830114210904412175296e-02
            2.54460438286207013547368660510983e-02 8.48608053360861092073719191830605e-01 1.25945902810518206571543942118296e-01 1.76480237095836232918966146598905e-02
            9.74553956171379409667565596464556e-01 1.78865986691015381448988108559206e-02 7.55944515951905218753559267952369e-03 6.29042122131170098643659915893522e-04
            8.70765592799697119019697311159689e-01 9.08417823768389587346661073752330e-02 3.83926248234639222456365814650781e-02 6.90111241800977874560896196953763e-03
            7.02922575688698647589092161069857e-01 2.08822428275864363511615806601185e-01 8.82549960354369888992920323289582e-02 2.16560808463863718742459241184406e-02
            4.99999999999999944488848768742173e-01 3.51461287844349379305697311792755e-01 1.48538712155650620694302688207245e-01 3.98973440527797540422483280053711e-02
            2.97077424311301407922059070187970e-01 4.94100147412834200810749507581932e-01 2.08822428275864335756040190972271e-01 5.12410128879843690397066779951274e-02
            1.29234407200302769958000226324657e-01 6.12080793311859716610001669323537e-01 2.58684799487837513431998104351806e-01 4.64988494614423861217566980030824e-02
            2.54460438286207013547368660510983e-02 6.85035977019597064341382974816952e-01 2.89517979151782234303880159131950e-01 2.40915834638253389954343219869770e-02
            9.74553956171379409667565596464556e-01 1.27230219143103211870693414198286e-02 1.27230219143102691453650621156157e-02 6.88562703523111063058648184664889e-04
            8.70765592799697119019697311159689e-01 6.46172036001513988567879209767852e-02 6.46172036001514821235147678635258e-02 7.55410243079215459188757719743990e-03
            7.02922575688698647589092161069857e-01 1.48538712155650648449878303836158e-01 1.48538712155650703961029535093985e-01 2.37052004161232461598718401774022e-02
            4.99999999999999944488848768742173e-01 2.50000000000000000000000000000000e-01 2.50000000000000000000000000000000e-01 4.36724698042482081805637506022322e-02
            2.97077424311301407922059070187970e-01 3.51461287844349212772243618019274e-01 3.51461287844349323794546080534928e-01 5.60894876894362098829205365291273e-02
            1.29234407200302769958000226324657e-01 4.35382796399848559509848655579844e-01 4.35382796399848670532151118095499e-01 5.08986161171708431649740589364228e-02
            2.54460438286207013547368660510983e-02 4.87276978085689593811480335716624e-01 4.87276978085689704833782798232278e-01 2.63711526754406185013568375552495e-02
            9.74553956171379409667565596464556e-01 7.55944515951910856604856192575426e-03 1.78865986691014826337475795980936e-02 6.29042122131169990223442667343079e-04
            8.70765592799697119019697311159689e-01 3.83926248234638667344853502072510e-02 9.08417823768390142458173386330600e-02 6.90111241800977701088548599273054e-03
            7.02922575688698647589092161069857e-01 8.82549960354370027770798401434149e-02 2.08822428275864335756040190972271e-01 2.16560808463863649353520202112122e-02
            4.99999999999999944488848768742173e-01 1.48538712155650731716605150722899e-01 3.51461287844349268283394849277101e-01 3.98973440527797401644605201909144e-02
            2.97077424311301407922059070187970e-01 2.08822428275864363511615806601185e-01 4.94100147412834145299598276324105e-01 5.12410128879843551619188701806706e-02
            1.29234407200302769958000226324657e-01 2.58684799487837568943149335609633e-01 6.12080793311859716610001669323537e-01 4.64988494614423722439688901886257e-02
            2.54460438286207013547368660510983e-02 2.89517979151782289815031390389777e-01 6.85035977019597064341382974816952e-01 2.40915834638253320565404180797486e-02
            9.74553956171379409667565596464556e-01 3.28850438978471170531348022336715e-03 2.21575394388358777597591853236736e-02 4.60797867536057788751674424432281e-04
            8.70765592799697119019697311159689e-01 1.67015320044136758581565516124101e-02 1.12532875195889198183252233320673e-01 5.05533377490165105466735084860375e-03
            7.02922575688698647589092161069857e-01 3.83926248234638459178036384855659e-02 2.58684799487837513431998104351806e-01 1.58639231334693243946620810902459e-02
            4.99999999999999944488848768742173e-01 6.46172036001513988567879209767852e-02 4.35382796399848615020999886837672e-01 2.92263592739814975773882110843260e-02
            2.97077424311301407922059070187970e-01 9.08417823768389171013026839318627e-02 6.12080793311859605587699206807883e-01 3.75360387459828109824044872766535e-02
            1.29234407200302769958000226324657e-01 1.12532875195889101038737578619475e-01 7.58232717603808170636625618499238e-01 3.40622192391829628488331138669309e-02
            2.54460438286207013547368660510983e-02 1.25945902810518040038090248344815e-01 8.48608053360861314118324116861913e-01 1.76480237095835955363209990309770e-02
            9.74553956171379409667565596464556e-01 6.47501146528084241781042518226741e-04 2.47985426820925072832757507512724e-02 2.13318720711471343943338441206947e-04
            8.70765592799697119019697311159689e-01 3.28850438978472037893086010740262e-03 1.25945902810518151060392710860469e-01 2.34028282161511530512498779899033e-03
            7.02922575688698647589092161069857e-01 7.55944515951912331119810772861456e-03 2.89517979151782234303880159131950e-01 7.34393977644001542948171135094526e-03
            4.99999999999999944488848768742173e-01 1.27230219143103541468153849791634e-02 4.87276978085689649322631566974451e-01 1.35298576894818702032186052974794e-02
            2.97077424311301407922059070187970e-01 1.78865986691015763088152823456767e-02 6.85035977019596953319080512301298e-01 1.73766858095164879183247563787518e-02
            1.29234407200302769958000226324657e-01 2.21575394388359818431677439320993e-02 8.48608053360861203096021654346259e-01 1.57685387555045429530675704654641e-02
            2.54460438286207013547368660510983e-02 2.47985426820926148361312613133123e-02 9.49755413489286715034154440218117e-01 8.16985951116516381786958334032533e-03
            ];
        
    case  12
        
        % ALG. DEG.:   12
        % PTS CARD.:   49
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.4e-15
        
        xyw_bar=[
            9.74553956171379409667565596464556e-01 2.47985426820925627944269820090994e-02 6.47501146528027538007421526344842e-04 2.13318720711471262628175504794115e-04
            8.70765592799697119019697311159689e-01 1.25945902810518123304817095231556e-01 3.28850438978475767548559360875515e-03 2.34028282161511443776324981058679e-03
            7.02922575688698647589092161069857e-01 2.89517979151782289815031390389777e-01 7.55944515951906259587644854036625e-03 7.34393977644001282739649738573462e-03
            4.99999999999999944488848768742173e-01 4.87276978085689815856085260747932e-01 1.27230219143101841439147392520681e-02 1.35298576894818649990481773670581e-02
            2.97077424311301407922059070187970e-01 6.85035977019597064341382974816952e-01 1.78865986691014722254067237372510e-02 1.73766858095164809794308524715234e-02
            1.29234407200302769958000226324657e-01 8.48608053360861314118324116861913e-01 2.21575394388359159236756568134297e-02 1.57685387555045360141736665582357e-02
            2.54460438286207013547368660510983e-02 9.49755413489286826056456902733771e-01 2.47985426820924725888062312151305e-02 8.16985951116516034842263138671115e-03
            9.74553956171379409667565596464556e-01 2.21575394388359298014634646278864e-02 3.28850438978466053097093890755787e-03 4.60797867536058493483086540010163e-04
            8.70765592799697119019697311159689e-01 1.12532875195889128794313194248389e-01 1.67015320044137521859894945919223e-02 5.05533377490165886092299274423567e-03
            7.02922575688698647589092161069857e-01 2.58684799487837457920846873093979e-01 3.83926248234638944900609658361645e-02 1.58639231334693486807907447655452e-02
            4.99999999999999944488848768742173e-01 4.35382796399848670532151118095499e-01 6.46172036001513294678488819045015e-02 2.92263592739815461496455384349247e-02
            2.97077424311301407922059070187970e-01 6.12080793311859494565396744292229e-01 9.08417823768390420013929542619735e-02 3.75360387459828664935557185344805e-02
            1.29234407200302769958000226324657e-01 7.58232717603808059614323155983584e-01 1.12532875195889170427676617691759e-01 3.40622192391830114210904412175296e-02
            2.54460438286207013547368660510983e-02 8.48608053360861092073719191830605e-01 1.25945902810518206571543942118296e-01 1.76480237095836232918966146598905e-02
            9.74553956171379409667565596464556e-01 1.78865986691015381448988108559206e-02 7.55944515951905218753559267952369e-03 6.29042122131170098643659915893522e-04
            8.70765592799697119019697311159689e-01 9.08417823768389587346661073752330e-02 3.83926248234639222456365814650781e-02 6.90111241800977874560896196953763e-03
            7.02922575688698647589092161069857e-01 2.08822428275864363511615806601185e-01 8.82549960354369888992920323289582e-02 2.16560808463863718742459241184406e-02
            4.99999999999999944488848768742173e-01 3.51461287844349379305697311792755e-01 1.48538712155650620694302688207245e-01 3.98973440527797540422483280053711e-02
            2.97077424311301407922059070187970e-01 4.94100147412834200810749507581932e-01 2.08822428275864335756040190972271e-01 5.12410128879843690397066779951274e-02
            1.29234407200302769958000226324657e-01 6.12080793311859716610001669323537e-01 2.58684799487837513431998104351806e-01 4.64988494614423861217566980030824e-02
            2.54460438286207013547368660510983e-02 6.85035977019597064341382974816952e-01 2.89517979151782234303880159131950e-01 2.40915834638253389954343219869770e-02
            9.74553956171379409667565596464556e-01 1.27230219143103211870693414198286e-02 1.27230219143102691453650621156157e-02 6.88562703523111063058648184664889e-04
            8.70765592799697119019697311159689e-01 6.46172036001513988567879209767852e-02 6.46172036001514821235147678635258e-02 7.55410243079215459188757719743990e-03
            7.02922575688698647589092161069857e-01 1.48538712155650648449878303836158e-01 1.48538712155650703961029535093985e-01 2.37052004161232461598718401774022e-02
            4.99999999999999944488848768742173e-01 2.50000000000000000000000000000000e-01 2.50000000000000000000000000000000e-01 4.36724698042482081805637506022322e-02
            2.97077424311301407922059070187970e-01 3.51461287844349212772243618019274e-01 3.51461287844349323794546080534928e-01 5.60894876894362098829205365291273e-02
            1.29234407200302769958000226324657e-01 4.35382796399848559509848655579844e-01 4.35382796399848670532151118095499e-01 5.08986161171708431649740589364228e-02
            2.54460438286207013547368660510983e-02 4.87276978085689593811480335716624e-01 4.87276978085689704833782798232278e-01 2.63711526754406185013568375552495e-02
            9.74553956171379409667565596464556e-01 7.55944515951910856604856192575426e-03 1.78865986691014826337475795980936e-02 6.29042122131169990223442667343079e-04
            8.70765592799697119019697311159689e-01 3.83926248234638667344853502072510e-02 9.08417823768390142458173386330600e-02 6.90111241800977701088548599273054e-03
            7.02922575688698647589092161069857e-01 8.82549960354370027770798401434149e-02 2.08822428275864335756040190972271e-01 2.16560808463863649353520202112122e-02
            4.99999999999999944488848768742173e-01 1.48538712155650731716605150722899e-01 3.51461287844349268283394849277101e-01 3.98973440527797401644605201909144e-02
            2.97077424311301407922059070187970e-01 2.08822428275864363511615806601185e-01 4.94100147412834145299598276324105e-01 5.12410128879843551619188701806706e-02
            1.29234407200302769958000226324657e-01 2.58684799487837568943149335609633e-01 6.12080793311859716610001669323537e-01 4.64988494614423722439688901886257e-02
            2.54460438286207013547368660510983e-02 2.89517979151782289815031390389777e-01 6.85035977019597064341382974816952e-01 2.40915834638253320565404180797486e-02
            9.74553956171379409667565596464556e-01 3.28850438978471170531348022336715e-03 2.21575394388358777597591853236736e-02 4.60797867536057788751674424432281e-04
            8.70765592799697119019697311159689e-01 1.67015320044136758581565516124101e-02 1.12532875195889198183252233320673e-01 5.05533377490165105466735084860375e-03
            7.02922575688698647589092161069857e-01 3.83926248234638459178036384855659e-02 2.58684799487837513431998104351806e-01 1.58639231334693243946620810902459e-02
            4.99999999999999944488848768742173e-01 6.46172036001513988567879209767852e-02 4.35382796399848615020999886837672e-01 2.92263592739814975773882110843260e-02
            2.97077424311301407922059070187970e-01 9.08417823768389171013026839318627e-02 6.12080793311859605587699206807883e-01 3.75360387459828109824044872766535e-02
            1.29234407200302769958000226324657e-01 1.12532875195889101038737578619475e-01 7.58232717603808170636625618499238e-01 3.40622192391829628488331138669309e-02
            2.54460438286207013547368660510983e-02 1.25945902810518040038090248344815e-01 8.48608053360861314118324116861913e-01 1.76480237095835955363209990309770e-02
            9.74553956171379409667565596464556e-01 6.47501146528084241781042518226741e-04 2.47985426820925072832757507512724e-02 2.13318720711471343943338441206947e-04
            8.70765592799697119019697311159689e-01 3.28850438978472037893086010740262e-03 1.25945902810518151060392710860469e-01 2.34028282161511530512498779899033e-03
            7.02922575688698647589092161069857e-01 7.55944515951912331119810772861456e-03 2.89517979151782234303880159131950e-01 7.34393977644001542948171135094526e-03
            4.99999999999999944488848768742173e-01 1.27230219143103541468153849791634e-02 4.87276978085689649322631566974451e-01 1.35298576894818702032186052974794e-02
            2.97077424311301407922059070187970e-01 1.78865986691015763088152823456767e-02 6.85035977019596953319080512301298e-01 1.73766858095164879183247563787518e-02
            1.29234407200302769958000226324657e-01 2.21575394388359818431677439320993e-02 8.48608053360861203096021654346259e-01 1.57685387555045429530675704654641e-02
            2.54460438286207013547368660510983e-02 2.47985426820926148361312613133123e-02 9.49755413489286715034154440218117e-01 8.16985951116516381786958334032533e-03
            ];
        
    case  13
        
        % ALG. DEG.:   13
        % PTS CARD.:   64
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.0e-15
        
        xyw_bar=[
            9.80144928248768199097185060963966e-01 1.94608478769852377698867940125638e-02 3.94223874246563132928145023470279e-04 1.01729610019562086912874032673670e-04
            8.98333238706813363982917053363053e-01 9.96481604529950576454311317320389e-02 2.01860084019157837165181490490795e-03 1.14432581808642072802861822111709e-03
            7.62766204958164717453428238513879e-01 2.32523501019462675021287623167154e-01 4.71029402237260752528413831896614e-03 3.76680585875795604505755775903708e-03
            5.91717321247824834529183135600761e-01 4.00176196870765610302811410292634e-01 8.10648188140955516800545410660561e-03 7.49483462621868077452313983144450e-03
            4.08282678752175054448514401883585e-01 5.79968731378002755327827344444813e-01 1.17485898698221902236582536716014e-02 1.08621396376050276255664783775501e-02
            2.37233795041835504591176686517429e-01 7.47621427229305357542443744023331e-01 1.51447777288591378663795694592409e-02 1.21112264346329764203025192159657e-02
            1.01666761293186524994780484121293e-01 8.80496767795773238596268583933124e-01 1.78364709110402364089509319455829e-02 1.01113274901409727624113088495506e-02
            1.98550717512318564139661702938611e-02 9.60684080371782989082873882580316e-01 1.94608478769852100143111783836503e-02 5.02187866972642147495031750281669e-03
            9.80144928248768199097185060963966e-01 1.78364709110402225311631241311261e-02 2.01860084019157837165181490490795e-03 2.23481804041844426839819037766688e-04
            8.98333238706813363982917053363053e-01 9.13306309413408484720875435414200e-02 1.03361303518457875449954030955269e-02 2.51387966776276923844268829100201e-03
            7.62766204958164717453428238513879e-01 2.13115003430640298498488505174464e-01 2.41187916111949840480832563116564e-02 8.27500045098709079149923439899794e-03
            5.91717321247824834529183135600761e-01 3.66773901111334954538989450156805e-01 4.15087776408402109318274142424343e-02 1.64648145504592766097395895030786e-02
            4.08282678752175054448514401883585e-01 5.31559337595478464955078834464075e-01 6.01579836523464805964067636523396e-02 2.38621829131125534617652306224045e-02
            2.37233795041835504591176686517429e-01 6.85218235276172871195399238786194e-01 7.75479696819916242134240746963769e-02 2.66062037616243730886456120288130e-02
            1.01666761293186524994780484121293e-01 8.07002607765472612655344164522830e-01 9.13306309413408623498753513558768e-02 2.22127825745136661672596289918147e-02
            1.98550717512318564139661702938611e-02 8.80496767795773127573966121417470e-01 9.96481604529950715232189395464957e-02 1.10321715041855200611919940456573e-02
            9.80144928248768199097185060963966e-01 1.51447777288591430705499973896622e-02 4.71029402237265783226494164637188e-03 3.15259470453653327121773530805626e-04
            8.98333238706813363982917053363053e-01 7.75479696819916103356362668819202e-02 2.41187916111950256814466797550267e-02 3.54625906230247433387203770394080e-03
            7.62766204958164717453428238513879e-01 1.80953921531883804574292184952355e-01 5.62798735099514779722795765337651e-02 1.16733094730766956415024537818681e-02
            5.91717321247824834529183135600761e-01 3.11424229421950138352315207157517e-01 9.68584493302250271185016572417226e-02 2.32264489654983273536625887345508e-02
            4.08282678752175054448514401883585e-01 4.51341975536214634612264262614190e-01 1.40375345711610310939221335502225e-01 3.36617076334658690828227634028735e-02
            2.37233795041835504591176686517429e-01 5.81812283426280774101257975416956e-01 1.80953921531883721307565338065615e-01 3.75326203609004649397462571869255e-02
            1.01666761293186524994780484121293e-01 6.85218235276173204262306626333157e-01 2.13115003430640270742912889545551e-01 3.13349451503090736803613935990143e-02
            1.98550717512318564139661702938611e-02 7.47621427229305579587048669054639e-01 2.32523501019462619510136391909327e-01 1.55627728229372781082462395829680e-02
            9.80144928248768199097185060963966e-01 1.17485898698222387959155810222001e-02 8.10648188140956210689935801383399e-03 3.64479041143735192718328130823124e-04
            8.98333238706813363982917053363053e-01 6.01579836523464389630433402089693e-02 4.15087776408401970540396064279776e-02 4.09991522479990003202665604931099e-03
            7.62766204958164717453428238513879e-01 1.40375345711610310939221335502225e-01 9.68584493302249716073504259838955e-02 1.34957932829065010288394077520024e-02
            5.91717321247824834529183135600761e-01 2.41587932983123310215489709662506e-01 1.66694745769051855255327154736733e-01 2.68526551666693306941180452440676e-02
            4.08282678752175054448514401883585e-01 3.50129388264701579824844657196081e-01 2.41587932983123365726640940920333e-01 3.89171081961519407799166003769642e-02
            2.37233795041835504591176686517429e-01 4.51341975536214412567659337582882e-01 3.11424229421950082841163975899690e-01 4.33923633160575514255974383104331e-02
            1.01666761293186524994780484121293e-01 5.31559337595478464955078834464075e-01 3.66773901111335010050140681414632e-01 3.62270822387719265700312121225579e-02
            1.98550717512318564139661702938611e-02 5.79968731378002644305524881929159e-01 4.00176196870765554791660179034807e-01 1.79924952226799272914092853170587e-02
            9.80144928248768199097185060963966e-01 8.10648188140955863745240606021980e-03 1.17485898698222422653625329758142e-02 3.64479041143735301138545379373568e-04
            8.98333238706813363982917053363053e-01 4.15087776408401831762517986135208e-02 6.01579836523464528408311480234261e-02 4.09991522479990089938839403771453e-03
            7.62766204958164717453428238513879e-01 9.68584493302249993629260416128091e-02 1.40375345711610283183645719873311e-01 1.34957932829065044982863597056166e-02
            5.91717321247824834529183135600761e-01 1.66694745769051827499751539107820e-01 2.41587932983123337971065325291420e-01 2.68526551666693341635649971976818e-02
            4.08282678752175054448514401883585e-01 2.41587932983123282459914094033593e-01 3.50129388264701635335995888453908e-01 3.89171081961519477188105042841926e-02
            2.37233795041835504591176686517429e-01 3.11424229421949971818861513384036e-01 4.51341975536214523589961800098536e-01 4.33923633160575583644913422176614e-02
            1.01666761293186524994780484121293e-01 3.66773901111334899027838218898978e-01 5.31559337595478575977381296979729e-01 3.62270822387719335089251160297863e-02
            1.98550717512318564139661702938611e-02 4.00176196870765499280508947776980e-01 5.79968731378002644305524881929159e-01 1.79924952226799307608562372706729e-02
            9.80144928248768199097185060963966e-01 4.71029402237266303643536957679316e-03 1.51447777288591378663795694592409e-02 3.15259470453652405549926918126857e-04
            8.98333238706813363982917053363053e-01 2.41187916111950534370222953839402e-02 7.75479696819915825800606512530067e-02 3.54625906230246435921205083730001e-03
            7.62766204958164717453428238513879e-01 5.62798735099515751167942312349624e-02 1.80953921531883721307565338065615e-01 1.16733094730766626817564102225333e-02
            5.91717321247824834529183135600761e-01 9.68584493302250965074406963140063e-02 3.11424229421950082841163975899690e-01 2.32264489654982614341705016158812e-02
            4.08282678752175054448514401883585e-01 1.40375345711610449717099413646793e-01 4.51341975536214468078810568840709e-01 3.36617076334657719383081087016762e-02
            2.37233795041835504591176686517429e-01 1.80953921531883887841019031839096e-01 5.81812283426280663078955512901302e-01 3.75326203609003539174437946712715e-02
            1.01666761293186524994780484121293e-01 2.13115003430640465031942198947945e-01 6.85218235276172982217701701301849e-01 3.13349451503089834747406428050454e-02
            1.98550717512318564139661702938611e-02 2.32523501019462841554741316940635e-01 7.47621427229305357542443744023331e-01 1.55627728229372347401593401627906e-02
            9.80144928248768199097185060963966e-01 2.01860084019157446852399395709199e-03 1.78364709110402260006100760847403e-02 2.23481804041844426839819037766688e-04
            8.98333238706813363982917053363053e-01 1.03361303518457806061014991882985e-02 9.13306309413408623498753513558768e-02 2.51387966776276923844268829100201e-03
            7.62766204958164717453428238513879e-01 2.41187916111950118036588719405700e-02 2.13115003430640270742912889545551e-01 8.27500045098709079149923439899794e-03
            5.91717321247824834529183135600761e-01 4.15087776408401484817822790773789e-02 3.66773901111335010050140681414632e-01 1.64648145504592766097395895030786e-02
            4.08282678752175054448514401883585e-01 6.01579836523463834518921089511423e-02 5.31559337595478575977381296979729e-01 2.38621829131125534617652306224045e-02
            2.37233795041835504591176686517429e-01 7.75479696819914993133338043662661e-02 6.85218235276172982217701701301849e-01 2.66062037616243730886456120288130e-02
            1.01666761293186524994780484121293e-01 9.13306309413407513275728888402227e-02 8.07002607765472723677646627038484e-01 2.22127825745136661672596289918147e-02
            1.98550717512318564139661702938611e-02 9.96481604529949466231286692163849e-02 8.80496767795773238596268583933124e-01 1.10321715041855200611919940456573e-02
            9.80144928248768199097185060963966e-01 3.94223874246564162920208884699491e-04 1.94608478769852377698867940125638e-02 1.01729610019561829414858067366367e-04
            8.98333238706813363982917053363053e-01 2.01860084019158227477963585272391e-03 9.96481604529950576454311317320389e-02 1.14432581808641790910296975880556e-03
            7.62766204958164717453428238513879e-01 4.71029402237267257741448744923218e-03 2.32523501019462619510136391909327e-01 3.76680585875794650407843988659806e-03
            5.91717321247824834529183135600761e-01 8.10648188140958465830454571232622e-03 4.00176196870765554791660179034807e-01 7.49483462621866255992664207497000e-03
            4.08282678752175054448514401883585e-01 1.17485898698222752251085765351490e-02 5.79968731378002644305524881929159e-01 1.08621396376050016047143387254437e-02
            2.37233795041835504591176686517429e-01 1.51447777288591794997429929026111e-02 7.47621427229305357542443744023331e-01 1.21112264346329469300034276102451e-02
            1.01666761293186524994780484121293e-01 1.78364709110402780423143553889531e-02 8.80496767795773238596268583933124e-01 1.01113274901409467415591691974441e-02
            1.98550717512318564139661702938611e-02 1.94608478769852932810380252703908e-02 9.60684080371782878060571420064662e-01 5.02187866972640933188598566516703e-03
            ];
        
    case  14
        
        % ALG. DEG.:   14
        % PTS CARD.:   64
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.0e-15
        
        xyw_bar=[
            9.80144928248768199097185060963966e-01 1.94608478769852377698867940125638e-02 3.94223874246563132928145023470279e-04 1.01729610019562086912874032673670e-04
            8.98333238706813363982917053363053e-01 9.96481604529950576454311317320389e-02 2.01860084019157837165181490490795e-03 1.14432581808642072802861822111709e-03
            7.62766204958164717453428238513879e-01 2.32523501019462675021287623167154e-01 4.71029402237260752528413831896614e-03 3.76680585875795604505755775903708e-03
            5.91717321247824834529183135600761e-01 4.00176196870765610302811410292634e-01 8.10648188140955516800545410660561e-03 7.49483462621868077452313983144450e-03
            4.08282678752175054448514401883585e-01 5.79968731378002755327827344444813e-01 1.17485898698221902236582536716014e-02 1.08621396376050276255664783775501e-02
            2.37233795041835504591176686517429e-01 7.47621427229305357542443744023331e-01 1.51447777288591378663795694592409e-02 1.21112264346329764203025192159657e-02
            1.01666761293186524994780484121293e-01 8.80496767795773238596268583933124e-01 1.78364709110402364089509319455829e-02 1.01113274901409727624113088495506e-02
            1.98550717512318564139661702938611e-02 9.60684080371782989082873882580316e-01 1.94608478769852100143111783836503e-02 5.02187866972642147495031750281669e-03
            9.80144928248768199097185060963966e-01 1.78364709110402225311631241311261e-02 2.01860084019157837165181490490795e-03 2.23481804041844426839819037766688e-04
            8.98333238706813363982917053363053e-01 9.13306309413408484720875435414200e-02 1.03361303518457875449954030955269e-02 2.51387966776276923844268829100201e-03
            7.62766204958164717453428238513879e-01 2.13115003430640298498488505174464e-01 2.41187916111949840480832563116564e-02 8.27500045098709079149923439899794e-03
            5.91717321247824834529183135600761e-01 3.66773901111334954538989450156805e-01 4.15087776408402109318274142424343e-02 1.64648145504592766097395895030786e-02
            4.08282678752175054448514401883585e-01 5.31559337595478464955078834464075e-01 6.01579836523464805964067636523396e-02 2.38621829131125534617652306224045e-02
            2.37233795041835504591176686517429e-01 6.85218235276172871195399238786194e-01 7.75479696819916242134240746963769e-02 2.66062037616243730886456120288130e-02
            1.01666761293186524994780484121293e-01 8.07002607765472612655344164522830e-01 9.13306309413408623498753513558768e-02 2.22127825745136661672596289918147e-02
            1.98550717512318564139661702938611e-02 8.80496767795773127573966121417470e-01 9.96481604529950715232189395464957e-02 1.10321715041855200611919940456573e-02
            9.80144928248768199097185060963966e-01 1.51447777288591430705499973896622e-02 4.71029402237265783226494164637188e-03 3.15259470453653327121773530805626e-04
            8.98333238706813363982917053363053e-01 7.75479696819916103356362668819202e-02 2.41187916111950256814466797550267e-02 3.54625906230247433387203770394080e-03
            7.62766204958164717453428238513879e-01 1.80953921531883804574292184952355e-01 5.62798735099514779722795765337651e-02 1.16733094730766956415024537818681e-02
            5.91717321247824834529183135600761e-01 3.11424229421950138352315207157517e-01 9.68584493302250271185016572417226e-02 2.32264489654983273536625887345508e-02
            4.08282678752175054448514401883585e-01 4.51341975536214634612264262614190e-01 1.40375345711610310939221335502225e-01 3.36617076334658690828227634028735e-02
            2.37233795041835504591176686517429e-01 5.81812283426280774101257975416956e-01 1.80953921531883721307565338065615e-01 3.75326203609004649397462571869255e-02
            1.01666761293186524994780484121293e-01 6.85218235276173204262306626333157e-01 2.13115003430640270742912889545551e-01 3.13349451503090736803613935990143e-02
            1.98550717512318564139661702938611e-02 7.47621427229305579587048669054639e-01 2.32523501019462619510136391909327e-01 1.55627728229372781082462395829680e-02
            9.80144928248768199097185060963966e-01 1.17485898698222387959155810222001e-02 8.10648188140956210689935801383399e-03 3.64479041143735192718328130823124e-04
            8.98333238706813363982917053363053e-01 6.01579836523464389630433402089693e-02 4.15087776408401970540396064279776e-02 4.09991522479990003202665604931099e-03
            7.62766204958164717453428238513879e-01 1.40375345711610310939221335502225e-01 9.68584493302249716073504259838955e-02 1.34957932829065010288394077520024e-02
            5.91717321247824834529183135600761e-01 2.41587932983123310215489709662506e-01 1.66694745769051855255327154736733e-01 2.68526551666693306941180452440676e-02
            4.08282678752175054448514401883585e-01 3.50129388264701579824844657196081e-01 2.41587932983123365726640940920333e-01 3.89171081961519407799166003769642e-02
            2.37233795041835504591176686517429e-01 4.51341975536214412567659337582882e-01 3.11424229421950082841163975899690e-01 4.33923633160575514255974383104331e-02
            1.01666761293186524994780484121293e-01 5.31559337595478464955078834464075e-01 3.66773901111335010050140681414632e-01 3.62270822387719265700312121225579e-02
            1.98550717512318564139661702938611e-02 5.79968731378002644305524881929159e-01 4.00176196870765554791660179034807e-01 1.79924952226799272914092853170587e-02
            9.80144928248768199097185060963966e-01 8.10648188140955863745240606021980e-03 1.17485898698222422653625329758142e-02 3.64479041143735301138545379373568e-04
            8.98333238706813363982917053363053e-01 4.15087776408401831762517986135208e-02 6.01579836523464528408311480234261e-02 4.09991522479990089938839403771453e-03
            7.62766204958164717453428238513879e-01 9.68584493302249993629260416128091e-02 1.40375345711610283183645719873311e-01 1.34957932829065044982863597056166e-02
            5.91717321247824834529183135600761e-01 1.66694745769051827499751539107820e-01 2.41587932983123337971065325291420e-01 2.68526551666693341635649971976818e-02
            4.08282678752175054448514401883585e-01 2.41587932983123282459914094033593e-01 3.50129388264701635335995888453908e-01 3.89171081961519477188105042841926e-02
            2.37233795041835504591176686517429e-01 3.11424229421949971818861513384036e-01 4.51341975536214523589961800098536e-01 4.33923633160575583644913422176614e-02
            1.01666761293186524994780484121293e-01 3.66773901111334899027838218898978e-01 5.31559337595478575977381296979729e-01 3.62270822387719335089251160297863e-02
            1.98550717512318564139661702938611e-02 4.00176196870765499280508947776980e-01 5.79968731378002644305524881929159e-01 1.79924952226799307608562372706729e-02
            9.80144928248768199097185060963966e-01 4.71029402237266303643536957679316e-03 1.51447777288591378663795694592409e-02 3.15259470453652405549926918126857e-04
            8.98333238706813363982917053363053e-01 2.41187916111950534370222953839402e-02 7.75479696819915825800606512530067e-02 3.54625906230246435921205083730001e-03
            7.62766204958164717453428238513879e-01 5.62798735099515751167942312349624e-02 1.80953921531883721307565338065615e-01 1.16733094730766626817564102225333e-02
            5.91717321247824834529183135600761e-01 9.68584493302250965074406963140063e-02 3.11424229421950082841163975899690e-01 2.32264489654982614341705016158812e-02
            4.08282678752175054448514401883585e-01 1.40375345711610449717099413646793e-01 4.51341975536214468078810568840709e-01 3.36617076334657719383081087016762e-02
            2.37233795041835504591176686517429e-01 1.80953921531883887841019031839096e-01 5.81812283426280663078955512901302e-01 3.75326203609003539174437946712715e-02
            1.01666761293186524994780484121293e-01 2.13115003430640465031942198947945e-01 6.85218235276172982217701701301849e-01 3.13349451503089834747406428050454e-02
            1.98550717512318564139661702938611e-02 2.32523501019462841554741316940635e-01 7.47621427229305357542443744023331e-01 1.55627728229372347401593401627906e-02
            9.80144928248768199097185060963966e-01 2.01860084019157446852399395709199e-03 1.78364709110402260006100760847403e-02 2.23481804041844426839819037766688e-04
            8.98333238706813363982917053363053e-01 1.03361303518457806061014991882985e-02 9.13306309413408623498753513558768e-02 2.51387966776276923844268829100201e-03
            7.62766204958164717453428238513879e-01 2.41187916111950118036588719405700e-02 2.13115003430640270742912889545551e-01 8.27500045098709079149923439899794e-03
            5.91717321247824834529183135600761e-01 4.15087776408401484817822790773789e-02 3.66773901111335010050140681414632e-01 1.64648145504592766097395895030786e-02
            4.08282678752175054448514401883585e-01 6.01579836523463834518921089511423e-02 5.31559337595478575977381296979729e-01 2.38621829131125534617652306224045e-02
            2.37233795041835504591176686517429e-01 7.75479696819914993133338043662661e-02 6.85218235276172982217701701301849e-01 2.66062037616243730886456120288130e-02
            1.01666761293186524994780484121293e-01 9.13306309413407513275728888402227e-02 8.07002607765472723677646627038484e-01 2.22127825745136661672596289918147e-02
            1.98550717512318564139661702938611e-02 9.96481604529949466231286692163849e-02 8.80496767795773238596268583933124e-01 1.10321715041855200611919940456573e-02
            9.80144928248768199097185060963966e-01 3.94223874246564162920208884699491e-04 1.94608478769852377698867940125638e-02 1.01729610019561829414858067366367e-04
            8.98333238706813363982917053363053e-01 2.01860084019158227477963585272391e-03 9.96481604529950576454311317320389e-02 1.14432581808641790910296975880556e-03
            7.62766204958164717453428238513879e-01 4.71029402237267257741448744923218e-03 2.32523501019462619510136391909327e-01 3.76680585875794650407843988659806e-03
            5.91717321247824834529183135600761e-01 8.10648188140958465830454571232622e-03 4.00176196870765554791660179034807e-01 7.49483462621866255992664207497000e-03
            4.08282678752175054448514401883585e-01 1.17485898698222752251085765351490e-02 5.79968731378002644305524881929159e-01 1.08621396376050016047143387254437e-02
            2.37233795041835504591176686517429e-01 1.51447777288591794997429929026111e-02 7.47621427229305357542443744023331e-01 1.21112264346329469300034276102451e-02
            1.01666761293186524994780484121293e-01 1.78364709110402780423143553889531e-02 8.80496767795773238596268583933124e-01 1.01113274901409467415591691974441e-02
            1.98550717512318564139661702938611e-02 1.94608478769852932810380252703908e-02 9.60684080371782878060571420064662e-01 5.02187866972640933188598566516703e-03
            ];
        
    case  15
        
        % ALG. DEG.:   15
        % PTS CARD.:   81
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.6e-15
        
        xyw_bar=[
            9.84080119753813153948840408702381e-01 1.56664376591339159217053378370110e-02 2.53442587052930129454253460608015e-04 5.25795930617662673296416886437044e-05
            9.18015553663317884769412557943724e-01 8.06792637689521530131031568089384e-02 1.30518256772996221748428524733754e-03 6.01850639556610273253611786969941e-04
            8.06685716350295178855844824283849e-01 1.90236743404124192258208836392441e-01 3.07754024558062888594633932370925e-03 2.04729249979192402808525486079816e-03
            6.62126711701904513418526221357752e-01 3.32494386010004416842633645501337e-01 5.37890228809106973884013314091135e-03 4.28859484157993828962185389741535e-03
            4.99999999999999944488848768742173e-01 4.92040059876906576974420204351190e-01 7.95994012309342302557979564880952e-03 6.71000039766201251262289773080738e-03
            3.37873288298095597603776241157902e-01 6.51585733743808570572753069427563e-01 1.05409779580958318234706894145347e-02 8.40431397989587283814394425007777e-03
            1.93314283649704710121852713200497e-01 7.93843376349689155979660881712334e-01 1.28423400006061338984864050871693e-02 8.54319497552427295494759107441496e-03
            8.19844463366822817640411358297570e-02 9.03400855984860751135556711233221e-01 1.46146976784569115892509216791950e-02 6.73918374500231433243957468448571e-03
            1.59198802461869015623108225554461e-02 9.68413682094679262313263734540669e-01 1.56664376591338916355766741617117e-02 3.25018350871220801368011166232463e-03
            9.84080119753813153948840408702381e-01 1.46146976784569341406561093776872e-02 1.30518256772991191050348191993180e-03 1.16868388285318292561798525497352e-04
            9.18015553663317884769412557943724e-01 7.52629968955497580251901013070892e-02 6.72144944113235720539734074918670e-03 1.33773028921776700531087733736513e-03
            8.06685716350295178855844824283849e-01 1.77465519135711513598607780295424e-01 1.58487645139933075455473954207264e-02 4.55050640118562135100210852556302e-03
            6.62126711701904513418526221357752e-01 3.10172933825021979181713049911195e-01 2.77003544730735073997607287310530e-02 9.53223747006574743945161998226467e-03
            4.99999999999999944488848768742173e-01 4.59007776831658942384706278971862e-01 4.09922231683410576152937210281380e-02 1.49142830175084221966397279857119e-02
            3.37873288298095597603776241157902e-01 6.07842619838295794565397045516875e-01 5.42840918636086078308267133252230e-02 1.86802250127799940315753701725043e-02
            1.93314283649704710121852713200497e-01 7.40550034527606482193107240163954e-01 6.61356818226888076850400466355495e-02 1.89889150801124811929199154292291e-02
            8.19844463366822817640411358297570e-02 8.42752556767767835310678492533043e-01 7.52629968955498274141291403793730e-02 1.49791486919997624688294379780018e-02
            1.59198802461869015623108225554461e-02 9.03400855984860973180161636264529e-01 8.06792637689521807686787724378519e-02 7.22416599627362637087157892779032e-03
            9.84080119753813153948840408702381e-01 1.28423400006061495109976888784331e-02 3.07754024558069654016190241918594e-03 1.68599292355775672806045273333098e-04
            9.18015553663317884769412557943724e-01 6.61356818226886966627375841198955e-02 1.58487645139934185678498579363804e-02 1.92986643723002684013001495344497e-03
            8.06685716350295178855844824283849e-01 1.55943871386706345294470565931988e-01 3.73704122629984758496846097841626e-02 6.56475348344223719188050836237380e-03
            6.62126711701904513418526221357752e-01 2.72557555606378953960700073366752e-01 6.53157326917165326207737052754965e-02 1.37516099571486204861114899244967e-02
            4.99999999999999944488848768742173e-01 4.03342858175147589427922412141925e-01 9.66571418248524105720775878580753e-02 2.15159770716336108675736937811962e-02
            3.37873288298095597603776241157902e-01 5.34128160743916113872842288401444e-01 1.27998550957988288523381470440654e-01 2.69488846762595685158103719913925e-02
            1.93314283649704710121852713200497e-01 6.50741844963589000094827952125343e-01 1.55943871386706289783319334674161e-01 2.73942140563692114463290039338972e-02
            8.19844463366822817640411358297570e-02 7.40550034527606260148502315132646e-01 1.77465519135711402576305317779770e-01 2.16095550440680433945672689333151e-02
            1.59198802461869015623108225554461e-02 7.93843376349689044957358419196680e-01 1.90236743404124108991481989505701e-01 1.04218881829604356759055860948138e-02
            9.84080119753813153948840408702381e-01 1.05409779580958023331715978088141e-02 5.37890228809104371798799348880493e-03 2.02069588413663665674505076630396e-04
            9.18015553663317884769412557943724e-01 5.42840918636085453807815781601676e-02 2.77003544730735698498058638961083e-02 2.31298311645052392937493834779161e-03
            8.06685716350295178855844824283849e-01 1.27998550957988344034532701698481e-01 6.53157326917164771096224740176694e-02 7.86798696424596648302252077655794e-03
            6.62126711701904513418526221357752e-01 2.23714929352727548428347859044152e-01 1.14158358945367938153125919598097e-01 1.64815766735397495335035955577041e-02
            4.99999999999999944488848768742173e-01 3.31063355850952256709263110678876e-01 1.68936644149047743290736889321124e-01 2.57873243145548061705518705366558e-02
            3.37873288298095597603776241157902e-01 4.38411782349176881723451515426859e-01 2.23714929352727520672772243415238e-01 3.22987715941769268357397493218741e-02
            1.93314283649704710121852713200497e-01 5.34128160743916335917447213432752e-01 2.72557555606378953960700073366752e-01 3.28325076691622086411115333248745e-02
            8.19844463366822817640411358297570e-02 6.07842619838295794565397045516875e-01 3.10172933825021868159410587395541e-01 2.58994793663951707041182004331858e-02
            1.59198802461869015623108225554461e-02 6.51585733743808681595055531943217e-01 3.32494386010004472353784876759164e-01 1.24908392330621616922448069431084e-02
            9.84080119753813153948840408702381e-01 7.95994012309342129085631967200243e-03 7.95994012309342476030327162561662e-03 2.13644805565290718876725328811972e-04
            9.18015553663317884769412557943724e-01 4.09922231683410229208242014919961e-02 4.09922231683410923097632405642798e-02 2.44547847139801573582151839048038e-03
            8.06685716350295178855844824283849e-01 9.66571418248524244498653956725320e-02 9.66571418248523966942897800436185e-02 8.31869138925265226569560894631650e-03
            6.62126711701904513418526221357752e-01 1.68936644149047715535161273692211e-01 1.68936644149047771046312504950038e-01 1.74256961251361246201252441778706e-02
            4.99999999999999944488848768742173e-01 2.49999999999999972244424384371086e-01 2.50000000000000000000000000000000e-01 2.72645078979119742657033498289820e-02
            3.37873288298095597603776241157902e-01 3.31063355850952145686960648163222e-01 3.31063355850952256709263110678876e-01 3.41489525039734495570620254056848e-02
            1.93314283649704710121852713200497e-01 4.03342858175147589427922412141925e-01 4.03342858175147700450224874657579e-01 3.47132627540145330402587831031269e-02
            8.19844463366822817640411358297570e-02 4.59007776831658775851252585198381e-01 4.59007776831658886873555047714035e-01 2.73830875636187631716467194564757e-02
            1.59198802461869015623108225554461e-02 4.92040059876906521463268973093363e-01 4.92040059876906632485571435609017e-01 1.32063559897587414621034085371321e-02
            9.84080119753813153948840408702381e-01 5.37890228809104545271146946561203e-03 1.05409779580958005984481218320070e-02 2.02069588413663530149233515942342e-04
            9.18015553663317884769412557943724e-01 2.77003544730735247469954884991239e-02 5.42840918636085939530389055107662e-02 2.31298311645052219465146237098452e-03
            8.06685716350295178855844824283849e-01 6.53157326917165603763493209044100e-02 1.27998550957988260767805854811741e-01 7.86798696424596127885209284613666e-03
            6.62126711701904513418526221357752e-01 1.14158358945367979786489343041467e-01 2.23714929352727520672772243415238e-01 1.64815766735397356557157877432473e-02
            4.99999999999999944488848768742173e-01 1.68936644149047798801888120578951e-01 3.31063355850952201198111879421049e-01 2.57873243145547853538701588149706e-02
            3.37873288298095597603776241157902e-01 2.23714929352727576183923474673065e-01 4.38411782349176826212300284169032e-01 3.22987715941769060190580376001890e-02
            1.93314283649704710121852713200497e-01 2.72557555606379064983002535882406e-01 5.34128160743916224895144750917098e-01 3.28325076691621878244298216031893e-02
            8.19844463366822817640411358297570e-02 3.10172933825021979181713049911195e-01 6.07842619838295683543094583001221e-01 2.58994793663951533568834406651149e-02
            1.59198802461869015623108225554461e-02 3.32494386010004527864936108016991e-01 6.51585733743808681595055531943217e-01 1.24908392330621530186274270590729e-02
            9.84080119753813153948840408702381e-01 3.07754024558069480543842644237884e-03 1.28423400006061512457211648552402e-02 1.68599292355775455965610776232211e-04
            9.18015553663317884769412557943724e-01 1.58487645139933491789108188640967e-02 6.61356818226887660516766231921792e-02 1.92986643723002445488523548533522e-03
            8.06685716350295178855844824283849e-01 3.73704122629985244219419371347612e-02 1.55943871386706289783319334674161e-01 6.56475348344222851826312847833833e-03
            6.62126711701904513418526221357752e-01 6.53157326917164909874102818321262e-02 2.72557555606379009471851304624579e-01 1.37516099571486031388767301564258e-02
            4.99999999999999944488848768742173e-01 9.66571418248523550609263566002483e-02 4.03342858175147644939073643399752e-01 2.15159770716335831119980781522827e-02
            3.37873288298095597603776241157902e-01 1.27998550957988205256654623553914e-01 5.34128160743916224895144750917098e-01 2.69488846762595338213408524552506e-02
            1.93314283649704710121852713200497e-01 1.55943871386706206516592487787420e-01 6.50741844963589111117130414640997e-01 2.73942140563691767518594843977553e-02
            8.19844463366822817640411358297570e-02 1.77465519135711319309578470893030e-01 7.40550034527606371170804777648300e-01 2.16095550440680156389916533044016e-02
            1.59198802461869015623108225554461e-02 1.90236743404124025724755142618960e-01 7.93843376349689155979660881712334e-01 1.04218881829604217981177782803570e-02
            9.84080119753813153948840408702381e-01 1.30518256772991386206739239383978e-03 1.46146976784569324059326334008801e-02 1.16868388285318048616309716258854e-04
            9.18015553663317884769412557943724e-01 6.72144944113231817411913127102707e-03 7.52629968955497996585535247504595e-02 1.33773028921776418638522887505360e-03
            8.06685716350295178855844824283849e-01 1.58487645139934081595090020755379e-02 1.77465519135711402576305317779770e-01 4.55050640118561181002299065312400e-03
            6.62126711701904513418526221357752e-01 2.77003544730735906664875756177935e-02 3.10172933825021868159410587395541e-01 9.53223747006572662276990826057954e-03
            4.99999999999999944488848768742173e-01 4.09922231683411408820205679148785e-02 4.59007776831658831362403816456208e-01 1.49142830175083892368936844263771e-02
            3.37873288298095597603776241157902e-01 5.42840918636086841586596563047351e-02 6.07842619838295683543094583001221e-01 1.86802250127799523982119467291341e-02
            1.93314283649704710121852713200497e-01 6.61356818226888909517668935222900e-02 7.40550034527606371170804777648300e-01 1.89889150801124395595564919858589e-02
            8.19844463366822817640411358297570e-02 7.52629968955499384364316028950270e-02 8.42752556767767724288376030017389e-01 1.49791486919997312438068703954741e-02
            1.59198802461869015623108225554461e-02 8.06792637689523611799202740257897e-02 9.03400855984860751135556711233221e-01 7.22416599627361075836029513652647e-03
            9.84080119753813153948840408702381e-01 2.53442587052931051026100073286784e-04 1.56664376591339159217053378370110e-02 5.25795930617662944346960007813152e-05
            9.18015553663317884769412557943724e-01 1.30518256772991472942913038224333e-03 8.06792637689521946464665802523086e-02 6.01850639556610598514263532621271e-04
            8.06685716350295178855844824283849e-01 3.07754024558070824954536526263382e-03 1.90236743404124108991481989505701e-01 2.04729249979192489544699284920171e-03
            6.62126711701904513418526221357752e-01 5.37890228809106193258449124527942e-03 3.32494386010004416842633645501337e-01 4.28859484157994089170706786262599e-03
            4.99999999999999944488848768742173e-01 7.95994012309345078115541127772303e-03 4.92040059876906576974420204351190e-01 6.71000039766201598206984968442157e-03
            3.37873288298095597603776241157902e-01 1.05409779580958370276411173449560e-02 6.51585733743808570572753069427563e-01 8.40431397989587630759089620369195e-03
            1.93314283649704710121852713200497e-01 1.28423400006061963485315402522247e-02 7.93843376349689044957358419196680e-01 8.54319497552427642439454302802915e-03
            8.19844463366822817640411358297570e-02 1.46146976784569827129134367282859e-02 9.03400855984860640113254248717567e-01 6.73918374500231780188652663809989e-03
            1.59198802461869015623108225554461e-02 1.56664376591339714328565690948381e-02 9.68413682094679151290961272025015e-01 3.25018350871220974840358763913173e-03
            ];
        
    case  16
        
        % ALG. DEG.:   16
        % PTS CARD.:   81
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.6e-15
        
        xyw_bar=[
            9.84080119753813153948840408702381e-01 1.56664376591339159217053378370110e-02 2.53442587052930129454253460608015e-04 5.25795930617662673296416886437044e-05
            9.18015553663317884769412557943724e-01 8.06792637689521530131031568089384e-02 1.30518256772996221748428524733754e-03 6.01850639556610273253611786969941e-04
            8.06685716350295178855844824283849e-01 1.90236743404124192258208836392441e-01 3.07754024558062888594633932370925e-03 2.04729249979192402808525486079816e-03
            6.62126711701904513418526221357752e-01 3.32494386010004416842633645501337e-01 5.37890228809106973884013314091135e-03 4.28859484157993828962185389741535e-03
            4.99999999999999944488848768742173e-01 4.92040059876906576974420204351190e-01 7.95994012309342302557979564880952e-03 6.71000039766201251262289773080738e-03
            3.37873288298095597603776241157902e-01 6.51585733743808570572753069427563e-01 1.05409779580958318234706894145347e-02 8.40431397989587283814394425007777e-03
            1.93314283649704710121852713200497e-01 7.93843376349689155979660881712334e-01 1.28423400006061338984864050871693e-02 8.54319497552427295494759107441496e-03
            8.19844463366822817640411358297570e-02 9.03400855984860751135556711233221e-01 1.46146976784569115892509216791950e-02 6.73918374500231433243957468448571e-03
            1.59198802461869015623108225554461e-02 9.68413682094679262313263734540669e-01 1.56664376591338916355766741617117e-02 3.25018350871220801368011166232463e-03
            9.84080119753813153948840408702381e-01 1.46146976784569341406561093776872e-02 1.30518256772991191050348191993180e-03 1.16868388285318292561798525497352e-04
            9.18015553663317884769412557943724e-01 7.52629968955497580251901013070892e-02 6.72144944113235720539734074918670e-03 1.33773028921776700531087733736513e-03
            8.06685716350295178855844824283849e-01 1.77465519135711513598607780295424e-01 1.58487645139933075455473954207264e-02 4.55050640118562135100210852556302e-03
            6.62126711701904513418526221357752e-01 3.10172933825021979181713049911195e-01 2.77003544730735073997607287310530e-02 9.53223747006574743945161998226467e-03
            4.99999999999999944488848768742173e-01 4.59007776831658942384706278971862e-01 4.09922231683410576152937210281380e-02 1.49142830175084221966397279857119e-02
            3.37873288298095597603776241157902e-01 6.07842619838295794565397045516875e-01 5.42840918636086078308267133252230e-02 1.86802250127799940315753701725043e-02
            1.93314283649704710121852713200497e-01 7.40550034527606482193107240163954e-01 6.61356818226888076850400466355495e-02 1.89889150801124811929199154292291e-02
            8.19844463366822817640411358297570e-02 8.42752556767767835310678492533043e-01 7.52629968955498274141291403793730e-02 1.49791486919997624688294379780018e-02
            1.59198802461869015623108225554461e-02 9.03400855984860973180161636264529e-01 8.06792637689521807686787724378519e-02 7.22416599627362637087157892779032e-03
            9.84080119753813153948840408702381e-01 1.28423400006061495109976888784331e-02 3.07754024558069654016190241918594e-03 1.68599292355775672806045273333098e-04
            9.18015553663317884769412557943724e-01 6.61356818226886966627375841198955e-02 1.58487645139934185678498579363804e-02 1.92986643723002684013001495344497e-03
            8.06685716350295178855844824283849e-01 1.55943871386706345294470565931988e-01 3.73704122629984758496846097841626e-02 6.56475348344223719188050836237380e-03
            6.62126711701904513418526221357752e-01 2.72557555606378953960700073366752e-01 6.53157326917165326207737052754965e-02 1.37516099571486204861114899244967e-02
            4.99999999999999944488848768742173e-01 4.03342858175147589427922412141925e-01 9.66571418248524105720775878580753e-02 2.15159770716336108675736937811962e-02
            3.37873288298095597603776241157902e-01 5.34128160743916113872842288401444e-01 1.27998550957988288523381470440654e-01 2.69488846762595685158103719913925e-02
            1.93314283649704710121852713200497e-01 6.50741844963589000094827952125343e-01 1.55943871386706289783319334674161e-01 2.73942140563692114463290039338972e-02
            8.19844463366822817640411358297570e-02 7.40550034527606260148502315132646e-01 1.77465519135711402576305317779770e-01 2.16095550440680433945672689333151e-02
            1.59198802461869015623108225554461e-02 7.93843376349689044957358419196680e-01 1.90236743404124108991481989505701e-01 1.04218881829604356759055860948138e-02
            9.84080119753813153948840408702381e-01 1.05409779580958023331715978088141e-02 5.37890228809104371798799348880493e-03 2.02069588413663665674505076630396e-04
            9.18015553663317884769412557943724e-01 5.42840918636085453807815781601676e-02 2.77003544730735698498058638961083e-02 2.31298311645052392937493834779161e-03
            8.06685716350295178855844824283849e-01 1.27998550957988344034532701698481e-01 6.53157326917164771096224740176694e-02 7.86798696424596648302252077655794e-03
            6.62126711701904513418526221357752e-01 2.23714929352727548428347859044152e-01 1.14158358945367938153125919598097e-01 1.64815766735397495335035955577041e-02
            4.99999999999999944488848768742173e-01 3.31063355850952256709263110678876e-01 1.68936644149047743290736889321124e-01 2.57873243145548061705518705366558e-02
            3.37873288298095597603776241157902e-01 4.38411782349176881723451515426859e-01 2.23714929352727520672772243415238e-01 3.22987715941769268357397493218741e-02
            1.93314283649704710121852713200497e-01 5.34128160743916335917447213432752e-01 2.72557555606378953960700073366752e-01 3.28325076691622086411115333248745e-02
            8.19844463366822817640411358297570e-02 6.07842619838295794565397045516875e-01 3.10172933825021868159410587395541e-01 2.58994793663951707041182004331858e-02
            1.59198802461869015623108225554461e-02 6.51585733743808681595055531943217e-01 3.32494386010004472353784876759164e-01 1.24908392330621616922448069431084e-02
            9.84080119753813153948840408702381e-01 7.95994012309342129085631967200243e-03 7.95994012309342476030327162561662e-03 2.13644805565290718876725328811972e-04
            9.18015553663317884769412557943724e-01 4.09922231683410229208242014919961e-02 4.09922231683410923097632405642798e-02 2.44547847139801573582151839048038e-03
            8.06685716350295178855844824283849e-01 9.66571418248524244498653956725320e-02 9.66571418248523966942897800436185e-02 8.31869138925265226569560894631650e-03
            6.62126711701904513418526221357752e-01 1.68936644149047715535161273692211e-01 1.68936644149047771046312504950038e-01 1.74256961251361246201252441778706e-02
            4.99999999999999944488848768742173e-01 2.49999999999999972244424384371086e-01 2.50000000000000000000000000000000e-01 2.72645078979119742657033498289820e-02
            3.37873288298095597603776241157902e-01 3.31063355850952145686960648163222e-01 3.31063355850952256709263110678876e-01 3.41489525039734495570620254056848e-02
            1.93314283649704710121852713200497e-01 4.03342858175147589427922412141925e-01 4.03342858175147700450224874657579e-01 3.47132627540145330402587831031269e-02
            8.19844463366822817640411358297570e-02 4.59007776831658775851252585198381e-01 4.59007776831658886873555047714035e-01 2.73830875636187631716467194564757e-02
            1.59198802461869015623108225554461e-02 4.92040059876906521463268973093363e-01 4.92040059876906632485571435609017e-01 1.32063559897587414621034085371321e-02
            9.84080119753813153948840408702381e-01 5.37890228809104545271146946561203e-03 1.05409779580958005984481218320070e-02 2.02069588413663530149233515942342e-04
            9.18015553663317884769412557943724e-01 2.77003544730735247469954884991239e-02 5.42840918636085939530389055107662e-02 2.31298311645052219465146237098452e-03
            8.06685716350295178855844824283849e-01 6.53157326917165603763493209044100e-02 1.27998550957988260767805854811741e-01 7.86798696424596127885209284613666e-03
            6.62126711701904513418526221357752e-01 1.14158358945367979786489343041467e-01 2.23714929352727520672772243415238e-01 1.64815766735397356557157877432473e-02
            4.99999999999999944488848768742173e-01 1.68936644149047798801888120578951e-01 3.31063355850952201198111879421049e-01 2.57873243145547853538701588149706e-02
            3.37873288298095597603776241157902e-01 2.23714929352727576183923474673065e-01 4.38411782349176826212300284169032e-01 3.22987715941769060190580376001890e-02
            1.93314283649704710121852713200497e-01 2.72557555606379064983002535882406e-01 5.34128160743916224895144750917098e-01 3.28325076691621878244298216031893e-02
            8.19844463366822817640411358297570e-02 3.10172933825021979181713049911195e-01 6.07842619838295683543094583001221e-01 2.58994793663951533568834406651149e-02
            1.59198802461869015623108225554461e-02 3.32494386010004527864936108016991e-01 6.51585733743808681595055531943217e-01 1.24908392330621530186274270590729e-02
            9.84080119753813153948840408702381e-01 3.07754024558069480543842644237884e-03 1.28423400006061512457211648552402e-02 1.68599292355775455965610776232211e-04
            9.18015553663317884769412557943724e-01 1.58487645139933491789108188640967e-02 6.61356818226887660516766231921792e-02 1.92986643723002445488523548533522e-03
            8.06685716350295178855844824283849e-01 3.73704122629985244219419371347612e-02 1.55943871386706289783319334674161e-01 6.56475348344222851826312847833833e-03
            6.62126711701904513418526221357752e-01 6.53157326917164909874102818321262e-02 2.72557555606379009471851304624579e-01 1.37516099571486031388767301564258e-02
            4.99999999999999944488848768742173e-01 9.66571418248523550609263566002483e-02 4.03342858175147644939073643399752e-01 2.15159770716335831119980781522827e-02
            3.37873288298095597603776241157902e-01 1.27998550957988205256654623553914e-01 5.34128160743916224895144750917098e-01 2.69488846762595338213408524552506e-02
            1.93314283649704710121852713200497e-01 1.55943871386706206516592487787420e-01 6.50741844963589111117130414640997e-01 2.73942140563691767518594843977553e-02
            8.19844463366822817640411358297570e-02 1.77465519135711319309578470893030e-01 7.40550034527606371170804777648300e-01 2.16095550440680156389916533044016e-02
            1.59198802461869015623108225554461e-02 1.90236743404124025724755142618960e-01 7.93843376349689155979660881712334e-01 1.04218881829604217981177782803570e-02
            9.84080119753813153948840408702381e-01 1.30518256772991386206739239383978e-03 1.46146976784569324059326334008801e-02 1.16868388285318048616309716258854e-04
            9.18015553663317884769412557943724e-01 6.72144944113231817411913127102707e-03 7.52629968955497996585535247504595e-02 1.33773028921776418638522887505360e-03
            8.06685716350295178855844824283849e-01 1.58487645139934081595090020755379e-02 1.77465519135711402576305317779770e-01 4.55050640118561181002299065312400e-03
            6.62126711701904513418526221357752e-01 2.77003544730735906664875756177935e-02 3.10172933825021868159410587395541e-01 9.53223747006572662276990826057954e-03
            4.99999999999999944488848768742173e-01 4.09922231683411408820205679148785e-02 4.59007776831658831362403816456208e-01 1.49142830175083892368936844263771e-02
            3.37873288298095597603776241157902e-01 5.42840918636086841586596563047351e-02 6.07842619838295683543094583001221e-01 1.86802250127799523982119467291341e-02
            1.93314283649704710121852713200497e-01 6.61356818226888909517668935222900e-02 7.40550034527606371170804777648300e-01 1.89889150801124395595564919858589e-02
            8.19844463366822817640411358297570e-02 7.52629968955499384364316028950270e-02 8.42752556767767724288376030017389e-01 1.49791486919997312438068703954741e-02
            1.59198802461869015623108225554461e-02 8.06792637689523611799202740257897e-02 9.03400855984860751135556711233221e-01 7.22416599627361075836029513652647e-03
            9.84080119753813153948840408702381e-01 2.53442587052931051026100073286784e-04 1.56664376591339159217053378370110e-02 5.25795930617662944346960007813152e-05
            9.18015553663317884769412557943724e-01 1.30518256772991472942913038224333e-03 8.06792637689521946464665802523086e-02 6.01850639556610598514263532621271e-04
            8.06685716350295178855844824283849e-01 3.07754024558070824954536526263382e-03 1.90236743404124108991481989505701e-01 2.04729249979192489544699284920171e-03
            6.62126711701904513418526221357752e-01 5.37890228809106193258449124527942e-03 3.32494386010004416842633645501337e-01 4.28859484157994089170706786262599e-03
            4.99999999999999944488848768742173e-01 7.95994012309345078115541127772303e-03 4.92040059876906576974420204351190e-01 6.71000039766201598206984968442157e-03
            3.37873288298095597603776241157902e-01 1.05409779580958370276411173449560e-02 6.51585733743808570572753069427563e-01 8.40431397989587630759089620369195e-03
            1.93314283649704710121852713200497e-01 1.28423400006061963485315402522247e-02 7.93843376349689044957358419196680e-01 8.54319497552427642439454302802915e-03
            8.19844463366822817640411358297570e-02 1.46146976784569827129134367282859e-02 9.03400855984860640113254248717567e-01 6.73918374500231780188652663809989e-03
            1.59198802461869015623108225554461e-02 1.56664376591339714328565690948381e-02 9.68413682094679151290961272025015e-01 3.25018350871220974840358763913173e-03
            ];
        
    case  17
        
        % ALG. DEG.:   17
        % PTS CARD.:  100
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 9.4e-16
        
        xyw_bar=[
            9.86953264258585760693165411794325e-01 1.28765184279079434453985086861394e-02 1.70217313506295861436079519535269e-04 2.89968147653883985082143986966585e-05
            9.32531683344492212661691610264825e-01 6.65880753571853195582619378001255e-02 8.80241298322467780046451935049845e-04 3.36131281175208425967410041934613e-04
            8.39704784149512217794608659460209e-01 1.58203886528673454137106091366149e-01 2.09132932181432806828524917364120e-03 1.17069866152235768988476571905721e-03
            7.16697697064623606699740321346326e-01 2.79606132654044436947060603415594e-01 3.69617028133195635319907523808070e-03 2.54297447049112038466867069530508e-03
            5.74437169490815580097375914192526e-01 4.20010624718162806079391202729312e-01 5.55220579102161382323288307816256e-03 4.19243165823399762531797918541088e-03
            4.25562830509184253369170392033993e-01 5.66942639540423010124925440322841e-01 7.49452995039268099475293638533913e-03 5.65906701052371671062379832051192e-03
            2.83302302935376448811410909911501e-01 7.07347131604541212723802345863078e-01 9.35056546008228295363551296759397e-03 6.43321260650272629816326386276160e-03
            1.60295215850487782205391340539791e-01 8.28749377729912195533756857912522e-01 1.09554064196000222608518015476875e-02 6.13269249279820276993246963570527e-03
            6.74683166555078983606108522508293e-02 9.20365188901400288479237588035176e-01 1.21664944430918131601515597139951e-02 4.64592989713297904730282539276232e-03
            1.30467357414141837956833569478476e-02 9.74076745830677936943686745507875e-01 1.28765184279079347717811288021039e-02 2.19353726119844499517497204976735e-03
            9.86953264258585760693165411794325e-01 1.21664944430917715267881362706248e-02 8.80241298322467780046451935049845e-04 6.49996356409456461098242407992132e-05
            9.32531683344492212661691610264825e-01 6.29163429031799198432040043371671e-02 4.55197375232786749510438539800816e-03 7.53476234568764471205559729583001e-04
            8.39704784149512217794608659460209e-01 1.49480367469124048040995944575116e-01 1.08148483813637341643953959646751e-02 2.62425328643770992484429704916238e-03
            7.16697697064623606699740321346326e-01 2.64188373451697822158479311838164e-01 1.91139294836785711417803668155102e-02 5.70036451808659562490166905490696e-03
            5.74437169490815580097375914192526e-01 3.96850822703576577854533979916596e-01 2.87120078056078420480901058908785e-02 9.39780911936743609813582622791728e-03
            4.25562830509184253369170392033993e-01 5.35680860640915690318308861606056e-01 3.87563088499000008013695151021238e-02 1.26854379257821271026474008181140e-02
            2.83302302935376448811410909911501e-01 6.68343309892794334992061067168834e-01 4.83543871718291606853767916618381e-02 1.44207727230282380798120911435944e-02
            1.60295215850487782205391340539791e-01 7.83051315875368136865120050060796e-01 5.66534682741440809294886093994137e-02 1.37471229428156316298981209911290e-02
            6.74683166555078983606108522508293e-02 8.69615340441312167918397335597547e-01 6.29163429031799337209918121516239e-02 1.04143766468956441850757244083070e-02
            1.30467357414141837956833569478476e-02 9.20365188901400510523842513066484e-01 6.65880753571853611916253612434957e-02 4.91706154266723614865153280106824e-03
            9.86953264258585760693165411794325e-01 1.09554064195999702191475222434747e-02 2.09132932181426908768706596219999e-03 9.52854144066027223310827443114590e-05
            9.32531683344492212661691610264825e-01 5.66534682741441086850642250283272e-02 1.08148483813636786532441647068481e-02 1.10454919552171819917274486755332e-03
            8.39704784149512217794608659460209e-01 1.34600659625933261054342438001186e-01 2.56945562245545211510489025386050e-02 3.84699174757510004019067650915531e-03
            7.16697697064623606699740321346326e-01 2.37890299135409960529230488646135e-01 4.54120037999664327710291900075390e-02 8.35637907841467984793037970803198e-03
            5.74437169490815580097375914192526e-01 3.57347144734770139073987138544908e-01 6.82156857744142808286369472625665e-02 1.37766024012753746791348774536345e-02
            4.25562830509184253369170392033993e-01 4.82357639414742134231772752173129e-01 9.20795300760735568879056245350512e-02 1.85960613127799899435554920046343e-02
            2.83302302935376448811410909911501e-01 6.01814485014102173998651323927334e-01 1.14883212050521321678786534903338e-01 2.11399539617048125295095672981915e-02
            1.60295215850487782205391340539791e-01 7.05104124523578956740266221459024e-01 1.34600659625933261054342438001186e-01 2.01524253726669995423481651641850e-02
            6.74683166555078983606108522508293e-02 7.83051315875368025842817587545142e-01 1.49480367469124075796571560204029e-01 1.52668270337316617363265791595950e-02
            1.30467357414141837956833569478476e-02 8.28749377729912306556059320428176e-01 1.58203886528673565159408553881804e-01 7.20810573991396714293999892220199e-03
            9.86953264258585760693165411794325e-01 9.35056546008230030087027273566491e-03 3.69617028133193900596431547000975e-03 1.17109940759036176079373181568855e-04
            9.32531683344492212661691610264825e-01 4.83543871718292161965280229196651e-02 1.91139294836785711417803668155102e-02 1.35753925885246534178973920603539e-03
            8.39704784149512217794608659460209e-01 1.14883212050521307800998727088881e-01 4.54120037999664744043926134509093e-02 4.72812107146382450129085128764928e-03
            7.16697697064623606699740321346326e-01 2.03042108086888606344189156516222e-01 8.02601948484877869560705221374519e-02 1.02703552786918474865052175459823e-02
            5.74437169490815580097375914192526e-01 3.04999900582235194956837176505360e-01 1.20562929926949224945786909302115e-01 1.69320467473598676444446198274818e-02
            4.25562830509184253369170392033993e-01 4.11697796482388467254054376098793e-01 1.62739373008427223865624000609387e-01 2.28553724854258649701321814973198e-02
            2.83302302935376448811410909911501e-01 5.13655588977734889333248702314449e-01 2.03042108086888606344189156516222e-01 2.59819277852923018190534065752217e-02
            1.60295215850487782205391340539791e-01 6.01814485014102285020953786442988e-01 2.37890299135409932773654873017222e-01 2.47682119686556366511265281360465e-02
            6.74683166555078983606108522508293e-02 6.68343309892794334992061067168834e-01 2.64188373451697766647328080580337e-01 1.87635979822623928658931191648662e-02
            1.30467357414141837956833569478476e-02 7.07347131604541434768407270894386e-01 2.79606132654044436947060603415594e-01 8.85907713623481155029715239379584e-03
            9.86953264258585760693165411794325e-01 7.49452995039265323917732075642562e-03 5.55220579102158606765726744924905e-03 1.28529899788176152786861594634615e-04
            9.32531683344492212661691610264825e-01 3.87563088498999452902182838442968e-02 2.87120078056078420480901058908785e-02 1.48991950442395825436603384872569e-03
            8.39704784149512217794608659460209e-01 9.20795300760734874989665854627674e-02 6.82156857744142947064247550770233e-02 5.18918311769975878350313891473888e-03
            7.16697697064623606699740321346326e-01 1.62739373008427196110048384980473e-01 1.20562929926949197190211293673201e-01 1.12718674965034907492489324454255e-02
            5.74437169490815580097375914192526e-01 2.44459107798195601768398432795948e-01 1.81103722710988818134225653011526e-01 1.85831728505844798182700117195054e-02
            4.25562830509184253369170392033993e-01 3.29978061692620061595704328283318e-01 2.44459107798195629523974048424861e-01 2.50841108460432733739953192753092e-02
            2.83302302935376448811410909911501e-01 4.11697796482388300720600682325312e-01 3.04999900582235194956837176505360e-01 2.85155517362822817639678163459394e-02
            1.60295215850487782205391340539791e-01 4.82357639414742078720621520915302e-01 3.57347144734770139073987138544908e-01 2.71834805963556032371020165783193e-02
            6.74683166555078983606108522508293e-02 5.35680860640915579296006399090402e-01 3.96850822703576522343382748658769e-01 2.05933275407256567712810380044175e-02
            1.30467357414141837956833569478476e-02 5.66942639540423010124925440322841e-01 4.20010624718162861590542433987139e-01 9.72296876896954972990805288191041e-03
            9.86953264258585760693165411794325e-01 5.55220579102158433293379147244195e-03 7.49452995039265497390079673323271e-03 1.28529899788175990156535721808950e-04
            9.32531683344492212661691610264825e-01 2.87120078056078351091962019836501e-02 3.87563088498999522291121877515252e-02 1.48991950442395651964255787191860e-03
            8.39704784149512217794608659460209e-01 6.82156857744142114396979081902828e-02 9.20795300760735707656934323495079e-02 5.18918311769975357933271098431760e-03
            7.16697697064623606699740321346326e-01 1.20562929926949155556847870229831e-01 1.62739373008427223865624000609387e-01 1.12718674965034768714611246309687e-02
            5.74437169490815580097375914192526e-01 1.81103722710988762623074421753699e-01 2.44459107798195657279549664053775e-01 1.85831728505844555321413480442061e-02
            4.25562830509184253369170392033993e-01 2.44459107798195546257247201538121e-01 3.29978061692620117106855559541145e-01 2.50841108460432456184197036463956e-02
            2.83302302935376448811410909911501e-01 3.04999900582235028423383482731879e-01 4.11697796482388467254054376098793e-01 2.85155517362822470694982968097975e-02
            1.60295215850487782205391340539791e-01 3.57347144734770028051684676029254e-01 4.82357639414742189742923983430956e-01 2.71834805963555720120794489957916e-02
            6.74683166555078983606108522508293e-02 3.96850822703576355809929054885288e-01 5.35680860640915801340611324121710e-01 2.05933275407256324851523743291182e-02
            1.30467357414141837956833569478476e-02 4.20010624718162695057088740213658e-01 5.66942639540423121147227902838495e-01 9.72296876896953932156719702106784e-03
            9.86953264258585760693165411794325e-01 3.69617028133193987332605345841330e-03 9.35056546008230030087027273566491e-03 1.17109940759036270947063274050493e-04
            9.32531683344492212661691610264825e-01 1.91139294836785711417803668155102e-02 4.83543871718292161965280229196651e-02 1.35753925885246642599191169153983e-03
            8.39704784149512217794608659460209e-01 4.54120037999664327710291900075390e-02 1.14883212050521349434362150532252e-01 4.72812107146382883809954122966701e-03
            7.16697697064623606699740321346326e-01 8.02601948484877869560705221374519e-02 2.03042108086888606344189156516222e-01 1.02703552786918561601225974300178e-02
            5.74437169490815580097375914192526e-01 1.20562929926949224945786909302115e-01 3.04999900582235194956837176505360e-01 1.69320467473598815222324276419386e-02
            4.25562830509184253369170392033993e-01 1.62739373008427251621199616238300e-01 4.11697796482388467254054376098793e-01 2.28553724854258823173669412653908e-02
            2.83302302935376448811410909911501e-01 2.03042108086888634099764772145136e-01 5.13655588977734889333248702314449e-01 2.59819277852923226357351182969069e-02
            1.60295215850487782205391340539791e-01 2.37890299135410016040381719903962e-01 6.01814485014102173998651323927334e-01 2.47682119686556574678082398577317e-02
            6.74683166555078983606108522508293e-02 2.64188373451697822158479311838164e-01 6.68343309892794223969758604653180e-01 1.87635979822624102131278789329372e-02
            1.30467357414141837956833569478476e-02 2.79606132654044547969363065931248e-01 7.07347131604541323746104808378732e-01 8.85907713623481848919105630102422e-03
            9.86953264258585760693165411794325e-01 2.09132932181426908768706596219999e-03 1.09554064195999702191475222434747e-02 9.52854144066026410159198078986265e-05
            9.32531683344492212661691610264825e-01 1.08148483813636803879676406836552e-02 5.66534682741441086850642250283272e-02 1.10454919552171733181100687914977e-03
            8.39704784149512217794608659460209e-01 2.56945562245544621704507193271638e-02 1.34600659625933316565493669259013e-01 3.84699174757509743810546254394467e-03
            7.16697697064623606699740321346326e-01 4.54120037999664397099230939147674e-02 2.37890299135409960529230488646135e-01 8.35637907841467464375995177761069e-03
            5.74437169490815580097375914192526e-01 6.82156857744142669508491394481098e-02 3.57347144734770139073987138544908e-01 1.37766024012753642707940215927920e-02
            4.25562830509184253369170392033993e-01 9.20795300760735291323300089061377e-02 4.82357639414742189742923983430956e-01 1.85960613127799760657676841901775e-02
            2.83302302935376448811410909911501e-01 1.14883212050521321678786534903338e-01 6.01814485014102173998651323927334e-01 2.11399539617047951822748075301206e-02
            1.60295215850487782205391340539791e-01 1.34600659625933316565493669259013e-01 7.05104124523578956740266221459024e-01 2.01524253726669821951134053961141e-02
            6.74683166555078983606108522508293e-02 1.49480367469124075796571560204029e-01 7.83051315875368025842817587545142e-01 1.52668270337316495932622473219453e-02
            1.30467357414141837956833569478476e-02 1.58203886528673537403832938252890e-01 8.28749377729912306556059320428176e-01 7.20810573991396193876957099178071e-03
            9.86953264258585760693165411794325e-01 8.80241298322468538987972674902949e-04 1.21664944430917715267881362706248e-02 6.49996356409457816350958014872674e-05
            9.32531683344492212661691610264825e-01 4.55197375232787703608350327044718e-03 6.29163429031799059654161965227104e-02 7.53476234568765989088601209289209e-04
            8.39704784149512217794608659460209e-01 1.08148483813636942657554484981119e-02 1.49480367469124075796571560204029e-01 2.62425328643771512901472497958366e-03
            7.16697697064623606699740321346326e-01 1.91139294836785988973559824444237e-02 2.64188373451697766647328080580337e-01 5.70036451808660690060426290415307e-03
            5.74437169490815580097375914192526e-01 2.87120078056078906203474332414771e-02 3.96850822703576522343382748658769e-01 9.39780911936745518009406197279532e-03
            4.25562830509184253369170392033993e-01 3.87563088499000146791573229165806e-02 5.35680860640915690318308861606056e-01 1.26854379257821531234995404702204e-02
            2.83302302935376448811410909911501e-01 4.83543871718292925243609658991772e-02 6.68343309892794223969758604653180e-01 1.44207727230282675701111827493150e-02
            1.60295215850487782205391340539791e-01 5.66534682741441988906849758222961e-02 7.83051315875368025842817587545142e-01 1.37471229428156593854737366200425e-02
            6.74683166555078983606108522508293e-02 6.29163429031800169877186590383644e-02 8.69615340441312056896094873081893e-01 1.04143766468956650017574361299921e-02
            1.30467357414141837956833569478476e-02 6.65880753571854305805644003157795e-02 9.20365188901400399501540050550830e-01 4.91706154266724568963065067350726e-03
            9.86953264258585760693165411794325e-01 1.70217313506295048284450155406944e-04 1.28765184279079434453985086861394e-02 2.89968147653884120607415547654639e-05
            9.32531683344492212661691610264825e-01 8.80241298322463334817544744481665e-04 6.65880753571853195582619378001255e-02 3.36131281175208588597735914760278e-04
            8.39704784149512217794608659460209e-01 2.09132932181425954670794808976098e-03 1.58203886528673509648257322623977e-01 1.17069866152235812356563471325899e-03
            7.16697697064623606699740321346326e-01 3.69617028133192382713390067294768e-03 2.79606132654044492458211834673421e-01 2.54297447049112125203040868370863e-03
            5.74437169490815580097375914192526e-01 5.55220579102156264889034176235327e-03 4.20010624718162861590542433987139e-01 4.19243165823399936004145516221797e-03
            4.25562830509184253369170392033993e-01 7.49452995039262201415475317389792e-03 5.66942639540423121147227902838495e-01 5.65906701052371931270901228572257e-03
            2.83302302935376448811410909911501e-01 9.35056546008225866750684929229465e-03 7.07347131604541212723802345863078e-01 6.43321260650272890024847782797224e-03
            1.60295215850487782205391340539791e-01 1.09554064195999233816136708696831e-02 8.28749377729912306556059320428176e-01 6.13269249279820537201768360091592e-03
            6.74683166555078983606108522508293e-02 1.21664944430917194850838569664120e-02 9.20365188901400399501540050550830e-01 4.64592989713298078202630136956941e-03
            1.30467357414141837956833569478476e-02 1.28765184279078896689707534051195e-02 9.74076745830677936943686745507875e-01 2.19353726119844586253671003817090e-03
            ];
        
    case  18
        
        % ALG. DEG.:   18
        % PTS CARD.:  100
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 9.4e-16
        
        xyw_bar=[
            9.86953264258585760693165411794325e-01 1.28765184279079434453985086861394e-02 1.70217313506295861436079519535269e-04 2.89968147653883985082143986966585e-05
            9.32531683344492212661691610264825e-01 6.65880753571853195582619378001255e-02 8.80241298322467780046451935049845e-04 3.36131281175208425967410041934613e-04
            8.39704784149512217794608659460209e-01 1.58203886528673454137106091366149e-01 2.09132932181432806828524917364120e-03 1.17069866152235768988476571905721e-03
            7.16697697064623606699740321346326e-01 2.79606132654044436947060603415594e-01 3.69617028133195635319907523808070e-03 2.54297447049112038466867069530508e-03
            5.74437169490815580097375914192526e-01 4.20010624718162806079391202729312e-01 5.55220579102161382323288307816256e-03 4.19243165823399762531797918541088e-03
            4.25562830509184253369170392033993e-01 5.66942639540423010124925440322841e-01 7.49452995039268099475293638533913e-03 5.65906701052371671062379832051192e-03
            2.83302302935376448811410909911501e-01 7.07347131604541212723802345863078e-01 9.35056546008228295363551296759397e-03 6.43321260650272629816326386276160e-03
            1.60295215850487782205391340539791e-01 8.28749377729912195533756857912522e-01 1.09554064196000222608518015476875e-02 6.13269249279820276993246963570527e-03
            6.74683166555078983606108522508293e-02 9.20365188901400288479237588035176e-01 1.21664944430918131601515597139951e-02 4.64592989713297904730282539276232e-03
            1.30467357414141837956833569478476e-02 9.74076745830677936943686745507875e-01 1.28765184279079347717811288021039e-02 2.19353726119844499517497204976735e-03
            9.86953264258585760693165411794325e-01 1.21664944430917715267881362706248e-02 8.80241298322467780046451935049845e-04 6.49996356409456461098242407992132e-05
            9.32531683344492212661691610264825e-01 6.29163429031799198432040043371671e-02 4.55197375232786749510438539800816e-03 7.53476234568764471205559729583001e-04
            8.39704784149512217794608659460209e-01 1.49480367469124048040995944575116e-01 1.08148483813637341643953959646751e-02 2.62425328643770992484429704916238e-03
            7.16697697064623606699740321346326e-01 2.64188373451697822158479311838164e-01 1.91139294836785711417803668155102e-02 5.70036451808659562490166905490696e-03
            5.74437169490815580097375914192526e-01 3.96850822703576577854533979916596e-01 2.87120078056078420480901058908785e-02 9.39780911936743609813582622791728e-03
            4.25562830509184253369170392033993e-01 5.35680860640915690318308861606056e-01 3.87563088499000008013695151021238e-02 1.26854379257821271026474008181140e-02
            2.83302302935376448811410909911501e-01 6.68343309892794334992061067168834e-01 4.83543871718291606853767916618381e-02 1.44207727230282380798120911435944e-02
            1.60295215850487782205391340539791e-01 7.83051315875368136865120050060796e-01 5.66534682741440809294886093994137e-02 1.37471229428156316298981209911290e-02
            6.74683166555078983606108522508293e-02 8.69615340441312167918397335597547e-01 6.29163429031799337209918121516239e-02 1.04143766468956441850757244083070e-02
            1.30467357414141837956833569478476e-02 9.20365188901400510523842513066484e-01 6.65880753571853611916253612434957e-02 4.91706154266723614865153280106824e-03
            9.86953264258585760693165411794325e-01 1.09554064195999702191475222434747e-02 2.09132932181426908768706596219999e-03 9.52854144066027223310827443114590e-05
            9.32531683344492212661691610264825e-01 5.66534682741441086850642250283272e-02 1.08148483813636786532441647068481e-02 1.10454919552171819917274486755332e-03
            8.39704784149512217794608659460209e-01 1.34600659625933261054342438001186e-01 2.56945562245545211510489025386050e-02 3.84699174757510004019067650915531e-03
            7.16697697064623606699740321346326e-01 2.37890299135409960529230488646135e-01 4.54120037999664327710291900075390e-02 8.35637907841467984793037970803198e-03
            5.74437169490815580097375914192526e-01 3.57347144734770139073987138544908e-01 6.82156857744142808286369472625665e-02 1.37766024012753746791348774536345e-02
            4.25562830509184253369170392033993e-01 4.82357639414742134231772752173129e-01 9.20795300760735568879056245350512e-02 1.85960613127799899435554920046343e-02
            2.83302302935376448811410909911501e-01 6.01814485014102173998651323927334e-01 1.14883212050521321678786534903338e-01 2.11399539617048125295095672981915e-02
            1.60295215850487782205391340539791e-01 7.05104124523578956740266221459024e-01 1.34600659625933261054342438001186e-01 2.01524253726669995423481651641850e-02
            6.74683166555078983606108522508293e-02 7.83051315875368025842817587545142e-01 1.49480367469124075796571560204029e-01 1.52668270337316617363265791595950e-02
            1.30467357414141837956833569478476e-02 8.28749377729912306556059320428176e-01 1.58203886528673565159408553881804e-01 7.20810573991396714293999892220199e-03
            9.86953264258585760693165411794325e-01 9.35056546008230030087027273566491e-03 3.69617028133193900596431547000975e-03 1.17109940759036176079373181568855e-04
            9.32531683344492212661691610264825e-01 4.83543871718292161965280229196651e-02 1.91139294836785711417803668155102e-02 1.35753925885246534178973920603539e-03
            8.39704784149512217794608659460209e-01 1.14883212050521307800998727088881e-01 4.54120037999664744043926134509093e-02 4.72812107146382450129085128764928e-03
            7.16697697064623606699740321346326e-01 2.03042108086888606344189156516222e-01 8.02601948484877869560705221374519e-02 1.02703552786918474865052175459823e-02
            5.74437169490815580097375914192526e-01 3.04999900582235194956837176505360e-01 1.20562929926949224945786909302115e-01 1.69320467473598676444446198274818e-02
            4.25562830509184253369170392033993e-01 4.11697796482388467254054376098793e-01 1.62739373008427223865624000609387e-01 2.28553724854258649701321814973198e-02
            2.83302302935376448811410909911501e-01 5.13655588977734889333248702314449e-01 2.03042108086888606344189156516222e-01 2.59819277852923018190534065752217e-02
            1.60295215850487782205391340539791e-01 6.01814485014102285020953786442988e-01 2.37890299135409932773654873017222e-01 2.47682119686556366511265281360465e-02
            6.74683166555078983606108522508293e-02 6.68343309892794334992061067168834e-01 2.64188373451697766647328080580337e-01 1.87635979822623928658931191648662e-02
            1.30467357414141837956833569478476e-02 7.07347131604541434768407270894386e-01 2.79606132654044436947060603415594e-01 8.85907713623481155029715239379584e-03
            9.86953264258585760693165411794325e-01 7.49452995039265323917732075642562e-03 5.55220579102158606765726744924905e-03 1.28529899788176152786861594634615e-04
            9.32531683344492212661691610264825e-01 3.87563088498999452902182838442968e-02 2.87120078056078420480901058908785e-02 1.48991950442395825436603384872569e-03
            8.39704784149512217794608659460209e-01 9.20795300760734874989665854627674e-02 6.82156857744142947064247550770233e-02 5.18918311769975878350313891473888e-03
            7.16697697064623606699740321346326e-01 1.62739373008427196110048384980473e-01 1.20562929926949197190211293673201e-01 1.12718674965034907492489324454255e-02
            5.74437169490815580097375914192526e-01 2.44459107798195601768398432795948e-01 1.81103722710988818134225653011526e-01 1.85831728505844798182700117195054e-02
            4.25562830509184253369170392033993e-01 3.29978061692620061595704328283318e-01 2.44459107798195629523974048424861e-01 2.50841108460432733739953192753092e-02
            2.83302302935376448811410909911501e-01 4.11697796482388300720600682325312e-01 3.04999900582235194956837176505360e-01 2.85155517362822817639678163459394e-02
            1.60295215850487782205391340539791e-01 4.82357639414742078720621520915302e-01 3.57347144734770139073987138544908e-01 2.71834805963556032371020165783193e-02
            6.74683166555078983606108522508293e-02 5.35680860640915579296006399090402e-01 3.96850822703576522343382748658769e-01 2.05933275407256567712810380044175e-02
            1.30467357414141837956833569478476e-02 5.66942639540423010124925440322841e-01 4.20010624718162861590542433987139e-01 9.72296876896954972990805288191041e-03
            9.86953264258585760693165411794325e-01 5.55220579102158433293379147244195e-03 7.49452995039265497390079673323271e-03 1.28529899788175990156535721808950e-04
            9.32531683344492212661691610264825e-01 2.87120078056078351091962019836501e-02 3.87563088498999522291121877515252e-02 1.48991950442395651964255787191860e-03
            8.39704784149512217794608659460209e-01 6.82156857744142114396979081902828e-02 9.20795300760735707656934323495079e-02 5.18918311769975357933271098431760e-03
            7.16697697064623606699740321346326e-01 1.20562929926949155556847870229831e-01 1.62739373008427223865624000609387e-01 1.12718674965034768714611246309687e-02
            5.74437169490815580097375914192526e-01 1.81103722710988762623074421753699e-01 2.44459107798195657279549664053775e-01 1.85831728505844555321413480442061e-02
            4.25562830509184253369170392033993e-01 2.44459107798195546257247201538121e-01 3.29978061692620117106855559541145e-01 2.50841108460432456184197036463956e-02
            2.83302302935376448811410909911501e-01 3.04999900582235028423383482731879e-01 4.11697796482388467254054376098793e-01 2.85155517362822470694982968097975e-02
            1.60295215850487782205391340539791e-01 3.57347144734770028051684676029254e-01 4.82357639414742189742923983430956e-01 2.71834805963555720120794489957916e-02
            6.74683166555078983606108522508293e-02 3.96850822703576355809929054885288e-01 5.35680860640915801340611324121710e-01 2.05933275407256324851523743291182e-02
            1.30467357414141837956833569478476e-02 4.20010624718162695057088740213658e-01 5.66942639540423121147227902838495e-01 9.72296876896953932156719702106784e-03
            9.86953264258585760693165411794325e-01 3.69617028133193987332605345841330e-03 9.35056546008230030087027273566491e-03 1.17109940759036270947063274050493e-04
            9.32531683344492212661691610264825e-01 1.91139294836785711417803668155102e-02 4.83543871718292161965280229196651e-02 1.35753925885246642599191169153983e-03
            8.39704784149512217794608659460209e-01 4.54120037999664327710291900075390e-02 1.14883212050521349434362150532252e-01 4.72812107146382883809954122966701e-03
            7.16697697064623606699740321346326e-01 8.02601948484877869560705221374519e-02 2.03042108086888606344189156516222e-01 1.02703552786918561601225974300178e-02
            5.74437169490815580097375914192526e-01 1.20562929926949224945786909302115e-01 3.04999900582235194956837176505360e-01 1.69320467473598815222324276419386e-02
            4.25562830509184253369170392033993e-01 1.62739373008427251621199616238300e-01 4.11697796482388467254054376098793e-01 2.28553724854258823173669412653908e-02
            2.83302302935376448811410909911501e-01 2.03042108086888634099764772145136e-01 5.13655588977734889333248702314449e-01 2.59819277852923226357351182969069e-02
            1.60295215850487782205391340539791e-01 2.37890299135410016040381719903962e-01 6.01814485014102173998651323927334e-01 2.47682119686556574678082398577317e-02
            6.74683166555078983606108522508293e-02 2.64188373451697822158479311838164e-01 6.68343309892794223969758604653180e-01 1.87635979822624102131278789329372e-02
            1.30467357414141837956833569478476e-02 2.79606132654044547969363065931248e-01 7.07347131604541323746104808378732e-01 8.85907713623481848919105630102422e-03
            9.86953264258585760693165411794325e-01 2.09132932181426908768706596219999e-03 1.09554064195999702191475222434747e-02 9.52854144066026410159198078986265e-05
            9.32531683344492212661691610264825e-01 1.08148483813636803879676406836552e-02 5.66534682741441086850642250283272e-02 1.10454919552171733181100687914977e-03
            8.39704784149512217794608659460209e-01 2.56945562245544621704507193271638e-02 1.34600659625933316565493669259013e-01 3.84699174757509743810546254394467e-03
            7.16697697064623606699740321346326e-01 4.54120037999664397099230939147674e-02 2.37890299135409960529230488646135e-01 8.35637907841467464375995177761069e-03
            5.74437169490815580097375914192526e-01 6.82156857744142669508491394481098e-02 3.57347144734770139073987138544908e-01 1.37766024012753642707940215927920e-02
            4.25562830509184253369170392033993e-01 9.20795300760735291323300089061377e-02 4.82357639414742189742923983430956e-01 1.85960613127799760657676841901775e-02
            2.83302302935376448811410909911501e-01 1.14883212050521321678786534903338e-01 6.01814485014102173998651323927334e-01 2.11399539617047951822748075301206e-02
            1.60295215850487782205391340539791e-01 1.34600659625933316565493669259013e-01 7.05104124523578956740266221459024e-01 2.01524253726669821951134053961141e-02
            6.74683166555078983606108522508293e-02 1.49480367469124075796571560204029e-01 7.83051315875368025842817587545142e-01 1.52668270337316495932622473219453e-02
            1.30467357414141837956833569478476e-02 1.58203886528673537403832938252890e-01 8.28749377729912306556059320428176e-01 7.20810573991396193876957099178071e-03
            9.86953264258585760693165411794325e-01 8.80241298322468538987972674902949e-04 1.21664944430917715267881362706248e-02 6.49996356409457816350958014872674e-05
            9.32531683344492212661691610264825e-01 4.55197375232787703608350327044718e-03 6.29163429031799059654161965227104e-02 7.53476234568765989088601209289209e-04
            8.39704784149512217794608659460209e-01 1.08148483813636942657554484981119e-02 1.49480367469124075796571560204029e-01 2.62425328643771512901472497958366e-03
            7.16697697064623606699740321346326e-01 1.91139294836785988973559824444237e-02 2.64188373451697766647328080580337e-01 5.70036451808660690060426290415307e-03
            5.74437169490815580097375914192526e-01 2.87120078056078906203474332414771e-02 3.96850822703576522343382748658769e-01 9.39780911936745518009406197279532e-03
            4.25562830509184253369170392033993e-01 3.87563088499000146791573229165806e-02 5.35680860640915690318308861606056e-01 1.26854379257821531234995404702204e-02
            2.83302302935376448811410909911501e-01 4.83543871718292925243609658991772e-02 6.68343309892794223969758604653180e-01 1.44207727230282675701111827493150e-02
            1.60295215850487782205391340539791e-01 5.66534682741441988906849758222961e-02 7.83051315875368025842817587545142e-01 1.37471229428156593854737366200425e-02
            6.74683166555078983606108522508293e-02 6.29163429031800169877186590383644e-02 8.69615340441312056896094873081893e-01 1.04143766468956650017574361299921e-02
            1.30467357414141837956833569478476e-02 6.65880753571854305805644003157795e-02 9.20365188901400399501540050550830e-01 4.91706154266724568963065067350726e-03
            9.86953264258585760693165411794325e-01 1.70217313506295048284450155406944e-04 1.28765184279079434453985086861394e-02 2.89968147653884120607415547654639e-05
            9.32531683344492212661691610264825e-01 8.80241298322463334817544744481665e-04 6.65880753571853195582619378001255e-02 3.36131281175208588597735914760278e-04
            8.39704784149512217794608659460209e-01 2.09132932181425954670794808976098e-03 1.58203886528673509648257322623977e-01 1.17069866152235812356563471325899e-03
            7.16697697064623606699740321346326e-01 3.69617028133192382713390067294768e-03 2.79606132654044492458211834673421e-01 2.54297447049112125203040868370863e-03
            5.74437169490815580097375914192526e-01 5.55220579102156264889034176235327e-03 4.20010624718162861590542433987139e-01 4.19243165823399936004145516221797e-03
            4.25562830509184253369170392033993e-01 7.49452995039262201415475317389792e-03 5.66942639540423121147227902838495e-01 5.65906701052371931270901228572257e-03
            2.83302302935376448811410909911501e-01 9.35056546008225866750684929229465e-03 7.07347131604541212723802345863078e-01 6.43321260650272890024847782797224e-03
            1.60295215850487782205391340539791e-01 1.09554064195999233816136708696831e-02 8.28749377729912306556059320428176e-01 6.13269249279820537201768360091592e-03
            6.74683166555078983606108522508293e-02 1.21664944430917194850838569664120e-02 9.20365188901400399501540050550830e-01 4.64592989713298078202630136956941e-03
            1.30467357414141837956833569478476e-02 1.28765184279078896689707534051195e-02 9.74076745830677936943686745507875e-01 2.19353726119844586253671003817090e-03
            ];
        
    case  19
        
        % ALG. DEG.:   19
        % PTS CARD.:  121
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.1e-15
        
        xyw_bar=[
            9.89114329073028430983072212256957e-01 1.07671730954412552677679215662465e-02 1.18497831530313749159866176796641e-04 1.68672892155245385010110120704852e-05
            9.43531299884047602866132820054190e-01 5.58540004288162958423491488701984e-02 6.14699687136101291518031075611361e-04 1.97382931280793029850742503761296e-04
            8.65076002787024678220006990159163e-01 1.33455258979163243759913370922732e-01 1.46873823381207802007963891810505e-03 6.99615271135270162632113777334553e-04
            7.59548064603405848060901917051524e-01 2.37834454754113411301119640484103e-01 2.61748064248074063797844246437307e-03 1.56072094208309169148518602554532e-03
            6.34771577976172451940328755881637e-01 3.61252665608499057281477462311159e-01 3.97575641532849077819378180720378e-03 2.67163721914268070811715460877167e-03
            5.00000000000000111022302462515654e-01 4.94557164536514104469233643612824e-01 5.44283546348578450846389387152158e-03 3.79833712774580280549518995769631e-03
            3.65228422023827603570822475376190e-01 6.27861663464529318190443518687971e-01 6.90991451164313374988523719366640e-03 4.64333899310944483301843987987922e-03
            2.40451935396594040916795620432822e-01 7.51279874318915186215406265546335e-01 8.26819028449077286779811402084306e-03 4.93006042554820170936702439234978e-03
            1.34923997212975321779993009840837e-01 8.55659070093865214978734456963139e-01 9.41693269315946324127253319602460e-03 4.48563928392313286674930594699617e-03
            5.64687001159524526450184112036368e-02 9.33260328644212044935102312592790e-01 1.02709712398354469087280449457467e-02 3.29805668173471355081316502833033e-03
            1.08856709269715135057765564852161e-02 9.78347155977587146225005199084990e-01 1.07671730954412847580670131719671e-02 1.53262739316846799929316969013371e-03
            9.89114329073028430983072212256957e-01 1.02709712398354694601332326442389e-02 6.14699687136099556794555098804267e-04 3.80502053723875903576585877363669e-05
            9.43531299884047602866132820054190e-01 5.32799860231670321941699342005450e-02 3.18871409278536493969724574526481e-03 4.45267818454518681057369144937752e-04
            8.65076002787024678220006990159163e-01 1.27305014475910222015997419475752e-01 7.61898273706509976399559036508435e-03 1.57823254277600962675043305694089e-03
            7.59548064603405848060901917051524e-01 2.26873927164383509813561090595613e-01 1.35780082322106421255369923528633e-02 3.52076445814361420338922670225656e-03
            6.34771577976172451940328755881637e-01 3.44604447786741541204236227713409e-01 2.06239742370860068554350164049538e-02 6.02683356939950117675541818584861e-03
            5.00000000000000111022302462515654e-01 4.71765649942023690410763947511441e-01 2.82343500579761985669335899729049e-02 8.56850830845246047573482428560965e-03
            3.65228422023827603570822475376190e-01 5.98926852097306117173047823598608e-01 3.58447258788663347672809322830290e-02 1.04747123289268499052306893304376e-02
            2.40451935396594040916795620432822e-01 7.16657372719664231830449807603145e-01 4.28906918837417272527545719640329e-02 1.11215150990434923777838704950227e-02
            1.34923997212975321779993009840837e-01 8.16226285408137353094559784949524e-01 4.88497173788873251254472052096389e-02 1.01189642152239005906677604684774e-02
            5.64687001159524526450184112036368e-02 8.90251313860880411077403095987393e-01 5.32799860231670807664272615511436e-02 7.43994677901646701834170727352102e-03
            1.08856709269715135057765564852161e-02 9.33260328644212155957404775108444e-01 5.58540004288162750256674371485133e-02 3.45738940764309977959900521682357e-03
            9.89114329073028430983072212256957e-01 9.41693269315949099684814882493811e-03 1.46873823381207802007963891810505e-03 5.64449747589463739260232100214409e-05
            9.43531299884047602866132820054190e-01 4.88497173788872973698715895807254e-02 7.61898273706509976399559036508435e-03 6.60525495924789971848023473199873e-04
            8.65076002787024678220006990159163e-01 1.16719512189048343908659433054709e-01 1.82044850239269778713335767861281e-02 2.34120407942360001657400303543000e-03
            7.59548064603405848060901917051524e-01 2.08009199135289563065143170206284e-01 3.24427362613045888739549127421924e-02 5.22282229562751575124845970776732e-03
            6.34771577976172451940328755881637e-01 3.15950343428585245231232647711295e-01 4.92780785952423028284385964070680e-02 8.94041084330103943966783219821082e-03
            5.00000000000000111022302462515654e-01 4.32538001393512228087701032563928e-01 6.74619986064876608899965049204184e-02 1.27108179958313107643652628553355e-02
            3.65228422023827603570822475376190e-01 5.49125659358439377477623111190042e-01 8.56459186177330744627056446915958e-02 1.55385461714894462098568439500923e-02
            2.40451935396594040916795620432822e-01 6.57066803651735198421590666839620e-01 1.02481260951670760661613712727558e-01 1.64980355008097108460152213638139e-02
            1.34923997212975321779993009840837e-01 7.48356490597976375944710980547825e-01 1.16719512189048302275296009611338e-01 1.50108172643262371970251933817053e-02
            5.64687001159524526450184112036368e-02 8.16226285408137242072257322433870e-01 1.27305014475910249771573035104666e-01 1.10366712620751806356311774948153e-02
            1.08856709269715135057765564852161e-02 8.55659070093865214978734456963139e-01 1.33455258979163216004337755293818e-01 5.12880958029945585585629430624977e-03
            9.89114329073028430983072212256957e-01 8.26819028449081797060848941782751e-03 2.61748064248075104631929832521564e-03 7.06565100269524978045948548910360e-05
            9.43531299884047602866132820054190e-01 4.28906918837417619472240915001748e-02 1.35780082322106351866430884456349e-02 8.26830493328736229781406041894343e-04
            8.65076002787024678220006990159163e-01 1.02481260951670732906038097098644e-01 3.24427362613045888739549127421924e-02 2.93066495678991978737770729424028e-03
            7.59548064603405848060901917051524e-01 1.82634802160626275036037213794771e-01 5.78171332359678769030608691537054e-02 6.53780779380199773831439102877994e-03
            6.34771577976172451940328755881637e-01 2.77408541086354154536763871874427e-01 8.78198809374733935229073722439352e-02 1.11913988994149088407326075866877e-02
            5.00000000000000111022302462515654e-01 3.79774032301702813008148496010108e-01 1.20225967698297075969549041474238e-01 1.59111071093335290638215440139902e-02
            3.65228422023827603570822475376190e-01 4.82139523517051693524138045177097e-01 1.52632054459120758416190710704541e-01 1.94507916436988809172348169340694e-02
            2.40451935396594040916795620432822e-01 5.76913262442779628536015934514580e-01 1.82634802160626330547188445052598e-01 2.06518581284903558759058483929039e-02
            1.34923997212975321779993009840837e-01 6.57066803651735087399288204323966e-01 2.08009199135289590820718785835197e-01 1.87901928396471545879009568125184e-02
            5.64687001159524526450184112036368e-02 7.16657372719664009785844882571837e-01 2.26873927164383482057985474966699e-01 1.38154490638583444389597687518290e-02
            1.08856709269715135057765564852161e-02 7.51279874318915075193103803030681e-01 2.37834454754113355789968409226276e-01 6.42012485760429414682892712562534e-03
            9.89114329073028430983072212256957e-01 6.90991451164308691235138581987485e-03 3.97575641532848210457640192316831e-03 7.96284238852016285114057692062772e-05
            9.43531299884047602866132820054190e-01 3.58447258788663764006443557263992e-02 2.06239742370860207332228242194105e-02 9.31820846782215614703681616504127e-04
            8.65076002787024678220006990159163e-01 8.56459186177330328293422212482255e-02 4.92780785952422889506507885926112e-02 3.30279872804013644241316605132397e-03
            7.59548064603405848060901917051524e-01 1.52632054459120758416190710704541e-01 8.78198809374733935229073722439352e-02 7.36797402088293102578875348740439e-03
            6.34771577976172451940328755881637e-01 2.31836621769812456506087983143516e-01 1.33391800254015091553583260974847e-01 1.26124748461401720717178775998946e-02
            5.00000000000000111022302462515654e-01 3.17385788988086170459013146682992e-01 1.82614211011913718518684390801354e-01 1.79314882790213812879809296418898e-02
            3.65228422023827603570822475376190e-01 4.02934956206359995434240772738121e-01 2.31836621769812456506087983143516e-01 2.19206394614788677155381435568415e-02
            2.40451935396594040916795620432822e-01 4.82139523517051804546440507692751e-01 2.77408541086354154536763871874427e-01 2.32742165222309810301837273982528e-02
            1.34923997212975321779993009840837e-01 5.49125659358439377477623111190042e-01 3.15950343428585300742383878969122e-01 2.11761582867502733129594361116688e-02
            5.64687001159524526450184112036368e-02 5.98926852097306006150745361082954e-01 3.44604447786741485693084996455582e-01 1.55697250515441712459541179214284e-02
            1.08856709269715135057765564852161e-02 6.27861663464529318190443518687971e-01 3.61252665608499112792628693568986e-01 7.23534778836690001951881257014065e-03
            9.89114329073028430983072212256957e-01 5.44283546348578537582563185992512e-03 5.44283546348578364110215588311803e-03 8.26948960846783740624710290489929e-05
            9.43531299884047602866132820054190e-01 2.82343500579762055058274938801333e-02 2.82343500579761916280396860656765e-02 9.67705052222097216949037168376435e-04
            8.65076002787024678220006990159163e-01 6.74619986064876747677843127348751e-02 6.74619986064876470122086971059616e-02 3.42998874368836102755064842995125e-03
            7.59548064603405848060901917051524e-01 1.20225967698297103725124657103152e-01 1.20225967698297048213973425845325e-01 7.65171299748351307207228089168893e-03
            6.34771577976172451940328755881637e-01 1.82614211011913801785411237688095e-01 1.82614211011913746274260006430268e-01 1.30981783373714782275909485065313e-02
            5.00000000000000111022302462515654e-01 2.50000000000000000000000000000000e-01 2.49999999999999888977697537484346e-01 1.86220257481810730348925631005841e-02
            3.65228422023827603570822475376190e-01 3.17385788988086281481315609198646e-01 3.17385788988086170459013146682992e-01 2.27647982206712652664126750323703e-02
            2.40451935396594040916795620432822e-01 3.79774032301703090563904652299243e-01 3.79774032301702868519299727267935e-01 2.41705012211836552105115316635420e-02
            1.34923997212975321779993009840837e-01 4.32538001393512450132305957595236e-01 4.32538001393512228087701032563928e-01 2.19916472479741685608356505099437e-02
            5.64687001159524526450184112036368e-02 4.71765649942023856944217641284922e-01 4.71765649942023634899612716253614e-01 1.61693115646828547427027444882697e-02
            1.08856709269715135057765564852161e-02 4.94557164536514326513838568644132e-01 4.94557164536514104469233643612824e-01 7.51397935940689969819272775453101e-03
            9.89114329073028430983072212256957e-01 3.97575641532848297193813991157185e-03 6.90991451164308604498964783147130e-03 7.96284238852015336437156767246393e-05
            9.43531299884047602866132820054190e-01 2.06239742370860242026697761730247e-02 3.58447258788663764006443557263992e-02 9.31820846782214638921726379550137e-04
            8.65076002787024678220006990159163e-01 4.92780785952422889506507885926112e-02 8.56459186177330328293422212482255e-02 3.30279872804013297296621409770978e-03
            7.59548064603405848060901917051524e-01 8.78198809374734212784829878728488e-02 1.52632054459120730660615095075627e-01 7.36797402088292321953311159177247e-03
            6.34771577976172451940328755881637e-01 1.33391800254015091553583260974847e-01 2.31836621769812456506087983143516e-01 1.26124748461401599286535457622449e-02
            5.00000000000000111022302462515654e-01 1.82614211011913774029835622059181e-01 3.17385788988086114947861915425165e-01 1.79314882790213604712992179202047e-02
            3.65228422023827603570822475376190e-01 2.31836621769812512017239214401343e-01 4.02934956206359939923089541480294e-01 2.19206394614788434294094798815422e-02
            2.40451935396594040916795620432822e-01 2.77408541086354210047915103132254e-01 4.82139523517051749035289276434924e-01 2.32742165222309567440550637229535e-02
            1.34923997212975321779993009840837e-01 3.15950343428585300742383878969122e-01 5.49125659358439377477623111190042e-01 2.11761582867502524962777243899836e-02
            5.64687001159524526450184112036368e-02 3.44604447786741541204236227713409e-01 5.98926852097305895128442898567300e-01 1.55697250515441538987193581533575e-02
            1.08856709269715135057765564852161e-02 3.61252665608499112792628693568986e-01 6.27861663464529318190443518687971e-01 7.23534778836689221326317067450873e-03
            9.89114329073028430983072212256957e-01 2.61748064248074974527669134261032e-03 8.26819028449081970533196539463461e-03 7.06565100269527959601922884047553e-05
            9.43531299884047602866132820054190e-01 1.35780082322106282477491845384066e-02 4.28906918837417688861179954074032e-02 8.26830493328739699228357995508532e-04
            8.65076002787024678220006990159163e-01 3.24427362613045749961671049277356e-02 1.02481260951670746783825904913101e-01 2.93066495678993236412290812609172e-03
            7.59548064603405848060901917051524e-01 5.78171332359678630252730613392487e-02 1.82634802160626275036037213794771e-01 6.53780779380202462652826866928990e-03
            6.34771577976172451940328755881637e-01 8.78198809374733657673317566150217e-02 2.77408541086354154536763871874427e-01 1.11913988994149556782664589604792e-02
            5.00000000000000111022302462515654e-01 1.20225967698296992702822194587498e-01 3.79774032301702924030450958525762e-01 1.59111071093335949833136311326598e-02
            3.65228422023827603570822475376190e-01 1.52632054459120675149463863817800e-01 4.82139523517051804546440507692751e-01 1.94507916436989641839616638208099e-02
            2.40451935396594040916795620432822e-01 1.82634802160626219524885982536944e-01 5.76913262442779739558318397030234e-01 2.06518581284904426120796472332586e-02
            1.34923997212975321779993009840837e-01 2.08009199135289452042840707690630e-01 6.57066803651735198421590666839620e-01 1.87901928396472309157338997920306e-02
            5.64687001159524526450184112036368e-02 2.26873927164383398791258628079959e-01 7.16657372719664120808147345087491e-01 1.38154490638584016848344759864631e-02
            1.08856709269715135057765564852161e-02 2.37834454754113300278817177968449e-01 7.51279874318915075193103803030681e-01 6.42012485760432103504280476613530e-03
            9.89114329073028430983072212256957e-01 1.46873823381207845376050791230682e-03 9.41693269315949099684814882493811e-03 5.64449747589461299805344007829433e-05
            9.43531299884047602866132820054190e-01 7.61898273706510063135732835348790e-03 4.88497173788872973698715895807254e-02 6.60525495924787152922375010888345e-04
            8.65076002787024678220006990159163e-01 1.82044850239269709324396728788997e-02 1.16719512189048357786447240869165e-01 2.34120407942359004191401616878920e-03
            7.59548064603405848060901917051524e-01 3.24427362613045888739549127421924e-02 2.08009199135289563065143170206284e-01 5.22282229562749319984327200927510e-03
            6.34771577976172451940328755881637e-01 4.92780785952422820117568846853828e-02 3.15950343428585245231232647711295e-01 8.94041084330100127575136070845474e-03
            5.00000000000000111022302462515654e-01 6.74619986064876470122086971059616e-02 4.32538001393512228087701032563928e-01 1.27108179958312552532140315975084e-02
            3.65228422023827603570822475376190e-01 8.56459186177330328293422212482255e-02 5.49125659358439377477623111190042e-01 1.55385461714893785556412808546156e-02
            2.40451935396594040916795620432822e-01 1.02481260951670746783825904913101e-01 6.57066803651735198421590666839620e-01 1.64980355008096379876292303379159e-02
            1.34923997212975321779993009840837e-01 1.16719512189048343908659433054709e-01 7.48356490597976375944710980547825e-01 1.50108172643261730122565822398428e-02
            5.64687001159524526450184112036368e-02 1.27305014475910194260421803846839e-01 8.16226285408137242072257322433870e-01 1.10366712620751337980973261210238e-02
            1.08856709269715135057765564852161e-02 1.33455258979163243759913370922732e-01 8.55659070093865214978734456963139e-01 5.12880958029943417181284459616109e-03
            9.89114329073028430983072212256957e-01 6.14699687136099665214772347354710e-04 1.02709712398354694601332326442389e-02 3.80502053723876581202943680803941e-05
            9.43531299884047602866132820054190e-01 3.18871409278536537337811473946658e-03 5.32799860231670321941699342005450e-02 4.45267818454519494208998509066078e-04
            8.65076002787024678220006990159163e-01 7.61898273706510757025123226071628e-03 1.27305014475910222015997419475752e-01 1.57823254277601244567608151925242e-03
            7.59548064603405848060901917051524e-01 1.35780082322106473297074202832846e-02 2.26873927164383509813561090595613e-01 3.52076445814362027492139262108140e-03
            6.34771577976172451940328755881637e-01 2.06239742370860415499045359410957e-02 3.44604447786741485693084996455582e-01 6.02683356939951158509627404669118e-03
            5.00000000000000111022302462515654e-01 2.82343500579762193836153016945900e-02 4.71765649942023690410763947511441e-01 8.56850830845247608824610807687350e-03
            3.65228422023827603570822475376190e-01 3.58447258788664110951138752625411e-02 5.98926852097306006150745361082954e-01 1.04747123289268672524654490985085e-02
            2.40451935396594040916795620432822e-01 4.28906918837418105194814188507735e-02 7.16657372719664120808147345087491e-01 1.11215150990435114597421062399007e-02
            1.34923997212975321779993009840837e-01 4.88497173788873459421289169313241e-02 8.16226285408137353094559784949524e-01 1.01189642152239179379025202365483e-02
            5.64687001159524526450184112036368e-02 5.32799860231670807664272615511436e-02 8.90251313860880411077403095987393e-01 7.43994677901648002876777709957423e-03
            1.08856709269715135057765564852161e-02 5.58540004288163513535003801280254e-02 9.33260328644212044935102312592790e-01 3.45738940764310585113117113564840e-03
            9.89114329073028430983072212256957e-01 1.18497831530313451004268743282921e-04 1.07671730954412552677679215662465e-02 1.68672892155244741265070207436594e-05
            9.43531299884047602866132820054190e-01 6.14699687136095978927385896639635e-04 5.58540004288163027812430527774268e-02 1.97382931280792243804167451770581e-04
            8.65076002787024678220006990159163e-01 1.46873823381207108118573501087667e-03 1.33455258979163243759913370922732e-01 6.99615271135267452126682563573468e-04
            7.59548064603405848060901917051524e-01 2.61748064248073760221235950496066e-03 2.37834454754113411301119640484103e-01 1.56072094208308561995302010672049e-03
            6.34771577976172451940328755881637e-01 3.97575641532846215525642818988672e-03 3.61252665608499112792628693568986e-01 2.67163721914266986609542975372733e-03
            5.00000000000000111022302462515654e-01 5.44283546348575588552654025420452e-03 4.94557164536514159980384874870651e-01 3.79833712774578762666477516063424e-03
            3.65228422023827603570822475376190e-01 6.90991451164305135052012829532941e-03 6.27861663464529429212745981203625e-01 4.64333899310942575106020413500119e-03
            2.40451935396594040916795620432822e-01 8.26819028449077633724506597445725e-03 7.51279874318915186215406265546335e-01 4.93006042554818176004705065906819e-03
            1.34923997212975321779993009840837e-01 9.41693269315944242459082147433946e-03 8.55659070093865214978734456963139e-01 4.48563928392311551951454617892523e-03
            5.64687001159524526450184112036368e-02 1.02709712398354156837054773632190e-02 9.33260328644212044935102312592790e-01 3.29805668173470010670622620807535e-03
            1.08856709269715135057765564852161e-02 1.07671730954411997566166903084195e-02 9.78347155977587257247307661600644e-01 1.53262739316846192776100377130888e-03
            ];
        
    case  20
        
        % ALG. DEG.:   20
        % PTS CARD.:  121
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.1e-15
        
        xyw_bar=[
            9.89114329073028430983072212256957e-01 1.07671730954412552677679215662465e-02 1.18497831530313749159866176796641e-04 1.68672892155245385010110120704852e-05
            9.43531299884047602866132820054190e-01 5.58540004288162958423491488701984e-02 6.14699687136101291518031075611361e-04 1.97382931280793029850742503761296e-04
            8.65076002787024678220006990159163e-01 1.33455258979163243759913370922732e-01 1.46873823381207802007963891810505e-03 6.99615271135270162632113777334553e-04
            7.59548064603405848060901917051524e-01 2.37834454754113411301119640484103e-01 2.61748064248074063797844246437307e-03 1.56072094208309169148518602554532e-03
            6.34771577976172451940328755881637e-01 3.61252665608499057281477462311159e-01 3.97575641532849077819378180720378e-03 2.67163721914268070811715460877167e-03
            5.00000000000000111022302462515654e-01 4.94557164536514104469233643612824e-01 5.44283546348578450846389387152158e-03 3.79833712774580280549518995769631e-03
            3.65228422023827603570822475376190e-01 6.27861663464529318190443518687971e-01 6.90991451164313374988523719366640e-03 4.64333899310944483301843987987922e-03
            2.40451935396594040916795620432822e-01 7.51279874318915186215406265546335e-01 8.26819028449077286779811402084306e-03 4.93006042554820170936702439234978e-03
            1.34923997212975321779993009840837e-01 8.55659070093865214978734456963139e-01 9.41693269315946324127253319602460e-03 4.48563928392313286674930594699617e-03
            5.64687001159524526450184112036368e-02 9.33260328644212044935102312592790e-01 1.02709712398354469087280449457467e-02 3.29805668173471355081316502833033e-03
            1.08856709269715135057765564852161e-02 9.78347155977587146225005199084990e-01 1.07671730954412847580670131719671e-02 1.53262739316846799929316969013371e-03
            9.89114329073028430983072212256957e-01 1.02709712398354694601332326442389e-02 6.14699687136099556794555098804267e-04 3.80502053723875903576585877363669e-05
            9.43531299884047602866132820054190e-01 5.32799860231670321941699342005450e-02 3.18871409278536493969724574526481e-03 4.45267818454518681057369144937752e-04
            8.65076002787024678220006990159163e-01 1.27305014475910222015997419475752e-01 7.61898273706509976399559036508435e-03 1.57823254277600962675043305694089e-03
            7.59548064603405848060901917051524e-01 2.26873927164383509813561090595613e-01 1.35780082322106421255369923528633e-02 3.52076445814361420338922670225656e-03
            6.34771577976172451940328755881637e-01 3.44604447786741541204236227713409e-01 2.06239742370860068554350164049538e-02 6.02683356939950117675541818584861e-03
            5.00000000000000111022302462515654e-01 4.71765649942023690410763947511441e-01 2.82343500579761985669335899729049e-02 8.56850830845246047573482428560965e-03
            3.65228422023827603570822475376190e-01 5.98926852097306117173047823598608e-01 3.58447258788663347672809322830290e-02 1.04747123289268499052306893304376e-02
            2.40451935396594040916795620432822e-01 7.16657372719664231830449807603145e-01 4.28906918837417272527545719640329e-02 1.11215150990434923777838704950227e-02
            1.34923997212975321779993009840837e-01 8.16226285408137353094559784949524e-01 4.88497173788873251254472052096389e-02 1.01189642152239005906677604684774e-02
            5.64687001159524526450184112036368e-02 8.90251313860880411077403095987393e-01 5.32799860231670807664272615511436e-02 7.43994677901646701834170727352102e-03
            1.08856709269715135057765564852161e-02 9.33260328644212155957404775108444e-01 5.58540004288162750256674371485133e-02 3.45738940764309977959900521682357e-03
            9.89114329073028430983072212256957e-01 9.41693269315949099684814882493811e-03 1.46873823381207802007963891810505e-03 5.64449747589463739260232100214409e-05
            9.43531299884047602866132820054190e-01 4.88497173788872973698715895807254e-02 7.61898273706509976399559036508435e-03 6.60525495924789971848023473199873e-04
            8.65076002787024678220006990159163e-01 1.16719512189048343908659433054709e-01 1.82044850239269778713335767861281e-02 2.34120407942360001657400303543000e-03
            7.59548064603405848060901917051524e-01 2.08009199135289563065143170206284e-01 3.24427362613045888739549127421924e-02 5.22282229562751575124845970776732e-03
            6.34771577976172451940328755881637e-01 3.15950343428585245231232647711295e-01 4.92780785952423028284385964070680e-02 8.94041084330103943966783219821082e-03
            5.00000000000000111022302462515654e-01 4.32538001393512228087701032563928e-01 6.74619986064876608899965049204184e-02 1.27108179958313107643652628553355e-02
            3.65228422023827603570822475376190e-01 5.49125659358439377477623111190042e-01 8.56459186177330744627056446915958e-02 1.55385461714894462098568439500923e-02
            2.40451935396594040916795620432822e-01 6.57066803651735198421590666839620e-01 1.02481260951670760661613712727558e-01 1.64980355008097108460152213638139e-02
            1.34923997212975321779993009840837e-01 7.48356490597976375944710980547825e-01 1.16719512189048302275296009611338e-01 1.50108172643262371970251933817053e-02
            5.64687001159524526450184112036368e-02 8.16226285408137242072257322433870e-01 1.27305014475910249771573035104666e-01 1.10366712620751806356311774948153e-02
            1.08856709269715135057765564852161e-02 8.55659070093865214978734456963139e-01 1.33455258979163216004337755293818e-01 5.12880958029945585585629430624977e-03
            9.89114329073028430983072212256957e-01 8.26819028449081797060848941782751e-03 2.61748064248075104631929832521564e-03 7.06565100269524978045948548910360e-05
            9.43531299884047602866132820054190e-01 4.28906918837417619472240915001748e-02 1.35780082322106351866430884456349e-02 8.26830493328736229781406041894343e-04
            8.65076002787024678220006990159163e-01 1.02481260951670732906038097098644e-01 3.24427362613045888739549127421924e-02 2.93066495678991978737770729424028e-03
            7.59548064603405848060901917051524e-01 1.82634802160626275036037213794771e-01 5.78171332359678769030608691537054e-02 6.53780779380199773831439102877994e-03
            6.34771577976172451940328755881637e-01 2.77408541086354154536763871874427e-01 8.78198809374733935229073722439352e-02 1.11913988994149088407326075866877e-02
            5.00000000000000111022302462515654e-01 3.79774032301702813008148496010108e-01 1.20225967698297075969549041474238e-01 1.59111071093335290638215440139902e-02
            3.65228422023827603570822475376190e-01 4.82139523517051693524138045177097e-01 1.52632054459120758416190710704541e-01 1.94507916436988809172348169340694e-02
            2.40451935396594040916795620432822e-01 5.76913262442779628536015934514580e-01 1.82634802160626330547188445052598e-01 2.06518581284903558759058483929039e-02
            1.34923997212975321779993009840837e-01 6.57066803651735087399288204323966e-01 2.08009199135289590820718785835197e-01 1.87901928396471545879009568125184e-02
            5.64687001159524526450184112036368e-02 7.16657372719664009785844882571837e-01 2.26873927164383482057985474966699e-01 1.38154490638583444389597687518290e-02
            1.08856709269715135057765564852161e-02 7.51279874318915075193103803030681e-01 2.37834454754113355789968409226276e-01 6.42012485760429414682892712562534e-03
            9.89114329073028430983072212256957e-01 6.90991451164308691235138581987485e-03 3.97575641532848210457640192316831e-03 7.96284238852016285114057692062772e-05
            9.43531299884047602866132820054190e-01 3.58447258788663764006443557263992e-02 2.06239742370860207332228242194105e-02 9.31820846782215614703681616504127e-04
            8.65076002787024678220006990159163e-01 8.56459186177330328293422212482255e-02 4.92780785952422889506507885926112e-02 3.30279872804013644241316605132397e-03
            7.59548064603405848060901917051524e-01 1.52632054459120758416190710704541e-01 8.78198809374733935229073722439352e-02 7.36797402088293102578875348740439e-03
            6.34771577976172451940328755881637e-01 2.31836621769812456506087983143516e-01 1.33391800254015091553583260974847e-01 1.26124748461401720717178775998946e-02
            5.00000000000000111022302462515654e-01 3.17385788988086170459013146682992e-01 1.82614211011913718518684390801354e-01 1.79314882790213812879809296418898e-02
            3.65228422023827603570822475376190e-01 4.02934956206359995434240772738121e-01 2.31836621769812456506087983143516e-01 2.19206394614788677155381435568415e-02
            2.40451935396594040916795620432822e-01 4.82139523517051804546440507692751e-01 2.77408541086354154536763871874427e-01 2.32742165222309810301837273982528e-02
            1.34923997212975321779993009840837e-01 5.49125659358439377477623111190042e-01 3.15950343428585300742383878969122e-01 2.11761582867502733129594361116688e-02
            5.64687001159524526450184112036368e-02 5.98926852097306006150745361082954e-01 3.44604447786741485693084996455582e-01 1.55697250515441712459541179214284e-02
            1.08856709269715135057765564852161e-02 6.27861663464529318190443518687971e-01 3.61252665608499112792628693568986e-01 7.23534778836690001951881257014065e-03
            9.89114329073028430983072212256957e-01 5.44283546348578537582563185992512e-03 5.44283546348578364110215588311803e-03 8.26948960846783740624710290489929e-05
            9.43531299884047602866132820054190e-01 2.82343500579762055058274938801333e-02 2.82343500579761916280396860656765e-02 9.67705052222097216949037168376435e-04
            8.65076002787024678220006990159163e-01 6.74619986064876747677843127348751e-02 6.74619986064876470122086971059616e-02 3.42998874368836102755064842995125e-03
            7.59548064603405848060901917051524e-01 1.20225967698297103725124657103152e-01 1.20225967698297048213973425845325e-01 7.65171299748351307207228089168893e-03
            6.34771577976172451940328755881637e-01 1.82614211011913801785411237688095e-01 1.82614211011913746274260006430268e-01 1.30981783373714782275909485065313e-02
            5.00000000000000111022302462515654e-01 2.50000000000000000000000000000000e-01 2.49999999999999888977697537484346e-01 1.86220257481810730348925631005841e-02
            3.65228422023827603570822475376190e-01 3.17385788988086281481315609198646e-01 3.17385788988086170459013146682992e-01 2.27647982206712652664126750323703e-02
            2.40451935396594040916795620432822e-01 3.79774032301703090563904652299243e-01 3.79774032301702868519299727267935e-01 2.41705012211836552105115316635420e-02
            1.34923997212975321779993009840837e-01 4.32538001393512450132305957595236e-01 4.32538001393512228087701032563928e-01 2.19916472479741685608356505099437e-02
            5.64687001159524526450184112036368e-02 4.71765649942023856944217641284922e-01 4.71765649942023634899612716253614e-01 1.61693115646828547427027444882697e-02
            1.08856709269715135057765564852161e-02 4.94557164536514326513838568644132e-01 4.94557164536514104469233643612824e-01 7.51397935940689969819272775453101e-03
            9.89114329073028430983072212256957e-01 3.97575641532848297193813991157185e-03 6.90991451164308604498964783147130e-03 7.96284238852015336437156767246393e-05
            9.43531299884047602866132820054190e-01 2.06239742370860242026697761730247e-02 3.58447258788663764006443557263992e-02 9.31820846782214638921726379550137e-04
            8.65076002787024678220006990159163e-01 4.92780785952422889506507885926112e-02 8.56459186177330328293422212482255e-02 3.30279872804013297296621409770978e-03
            7.59548064603405848060901917051524e-01 8.78198809374734212784829878728488e-02 1.52632054459120730660615095075627e-01 7.36797402088292321953311159177247e-03
            6.34771577976172451940328755881637e-01 1.33391800254015091553583260974847e-01 2.31836621769812456506087983143516e-01 1.26124748461401599286535457622449e-02
            5.00000000000000111022302462515654e-01 1.82614211011913774029835622059181e-01 3.17385788988086114947861915425165e-01 1.79314882790213604712992179202047e-02
            3.65228422023827603570822475376190e-01 2.31836621769812512017239214401343e-01 4.02934956206359939923089541480294e-01 2.19206394614788434294094798815422e-02
            2.40451935396594040916795620432822e-01 2.77408541086354210047915103132254e-01 4.82139523517051749035289276434924e-01 2.32742165222309567440550637229535e-02
            1.34923997212975321779993009840837e-01 3.15950343428585300742383878969122e-01 5.49125659358439377477623111190042e-01 2.11761582867502524962777243899836e-02
            5.64687001159524526450184112036368e-02 3.44604447786741541204236227713409e-01 5.98926852097305895128442898567300e-01 1.55697250515441538987193581533575e-02
            1.08856709269715135057765564852161e-02 3.61252665608499112792628693568986e-01 6.27861663464529318190443518687971e-01 7.23534778836689221326317067450873e-03
            9.89114329073028430983072212256957e-01 2.61748064248074974527669134261032e-03 8.26819028449081970533196539463461e-03 7.06565100269527959601922884047553e-05
            9.43531299884047602866132820054190e-01 1.35780082322106282477491845384066e-02 4.28906918837417688861179954074032e-02 8.26830493328739699228357995508532e-04
            8.65076002787024678220006990159163e-01 3.24427362613045749961671049277356e-02 1.02481260951670746783825904913101e-01 2.93066495678993236412290812609172e-03
            7.59548064603405848060901917051524e-01 5.78171332359678630252730613392487e-02 1.82634802160626275036037213794771e-01 6.53780779380202462652826866928990e-03
            6.34771577976172451940328755881637e-01 8.78198809374733657673317566150217e-02 2.77408541086354154536763871874427e-01 1.11913988994149556782664589604792e-02
            5.00000000000000111022302462515654e-01 1.20225967698296992702822194587498e-01 3.79774032301702924030450958525762e-01 1.59111071093335949833136311326598e-02
            3.65228422023827603570822475376190e-01 1.52632054459120675149463863817800e-01 4.82139523517051804546440507692751e-01 1.94507916436989641839616638208099e-02
            2.40451935396594040916795620432822e-01 1.82634802160626219524885982536944e-01 5.76913262442779739558318397030234e-01 2.06518581284904426120796472332586e-02
            1.34923997212975321779993009840837e-01 2.08009199135289452042840707690630e-01 6.57066803651735198421590666839620e-01 1.87901928396472309157338997920306e-02
            5.64687001159524526450184112036368e-02 2.26873927164383398791258628079959e-01 7.16657372719664120808147345087491e-01 1.38154490638584016848344759864631e-02
            1.08856709269715135057765564852161e-02 2.37834454754113300278817177968449e-01 7.51279874318915075193103803030681e-01 6.42012485760432103504280476613530e-03
            9.89114329073028430983072212256957e-01 1.46873823381207845376050791230682e-03 9.41693269315949099684814882493811e-03 5.64449747589461299805344007829433e-05
            9.43531299884047602866132820054190e-01 7.61898273706510063135732835348790e-03 4.88497173788872973698715895807254e-02 6.60525495924787152922375010888345e-04
            8.65076002787024678220006990159163e-01 1.82044850239269709324396728788997e-02 1.16719512189048357786447240869165e-01 2.34120407942359004191401616878920e-03
            7.59548064603405848060901917051524e-01 3.24427362613045888739549127421924e-02 2.08009199135289563065143170206284e-01 5.22282229562749319984327200927510e-03
            6.34771577976172451940328755881637e-01 4.92780785952422820117568846853828e-02 3.15950343428585245231232647711295e-01 8.94041084330100127575136070845474e-03
            5.00000000000000111022302462515654e-01 6.74619986064876470122086971059616e-02 4.32538001393512228087701032563928e-01 1.27108179958312552532140315975084e-02
            3.65228422023827603570822475376190e-01 8.56459186177330328293422212482255e-02 5.49125659358439377477623111190042e-01 1.55385461714893785556412808546156e-02
            2.40451935396594040916795620432822e-01 1.02481260951670746783825904913101e-01 6.57066803651735198421590666839620e-01 1.64980355008096379876292303379159e-02
            1.34923997212975321779993009840837e-01 1.16719512189048343908659433054709e-01 7.48356490597976375944710980547825e-01 1.50108172643261730122565822398428e-02
            5.64687001159524526450184112036368e-02 1.27305014475910194260421803846839e-01 8.16226285408137242072257322433870e-01 1.10366712620751337980973261210238e-02
            1.08856709269715135057765564852161e-02 1.33455258979163243759913370922732e-01 8.55659070093865214978734456963139e-01 5.12880958029943417181284459616109e-03
            9.89114329073028430983072212256957e-01 6.14699687136099665214772347354710e-04 1.02709712398354694601332326442389e-02 3.80502053723876581202943680803941e-05
            9.43531299884047602866132820054190e-01 3.18871409278536537337811473946658e-03 5.32799860231670321941699342005450e-02 4.45267818454519494208998509066078e-04
            8.65076002787024678220006990159163e-01 7.61898273706510757025123226071628e-03 1.27305014475910222015997419475752e-01 1.57823254277601244567608151925242e-03
            7.59548064603405848060901917051524e-01 1.35780082322106473297074202832846e-02 2.26873927164383509813561090595613e-01 3.52076445814362027492139262108140e-03
            6.34771577976172451940328755881637e-01 2.06239742370860415499045359410957e-02 3.44604447786741485693084996455582e-01 6.02683356939951158509627404669118e-03
            5.00000000000000111022302462515654e-01 2.82343500579762193836153016945900e-02 4.71765649942023690410763947511441e-01 8.56850830845247608824610807687350e-03
            3.65228422023827603570822475376190e-01 3.58447258788664110951138752625411e-02 5.98926852097306006150745361082954e-01 1.04747123289268672524654490985085e-02
            2.40451935396594040916795620432822e-01 4.28906918837418105194814188507735e-02 7.16657372719664120808147345087491e-01 1.11215150990435114597421062399007e-02
            1.34923997212975321779993009840837e-01 4.88497173788873459421289169313241e-02 8.16226285408137353094559784949524e-01 1.01189642152239179379025202365483e-02
            5.64687001159524526450184112036368e-02 5.32799860231670807664272615511436e-02 8.90251313860880411077403095987393e-01 7.43994677901648002876777709957423e-03
            1.08856709269715135057765564852161e-02 5.58540004288163513535003801280254e-02 9.33260328644212044935102312592790e-01 3.45738940764310585113117113564840e-03
            9.89114329073028430983072212256957e-01 1.18497831530313451004268743282921e-04 1.07671730954412552677679215662465e-02 1.68672892155244741265070207436594e-05
            9.43531299884047602866132820054190e-01 6.14699687136095978927385896639635e-04 5.58540004288163027812430527774268e-02 1.97382931280792243804167451770581e-04
            8.65076002787024678220006990159163e-01 1.46873823381207108118573501087667e-03 1.33455258979163243759913370922732e-01 6.99615271135267452126682563573468e-04
            7.59548064603405848060901917051524e-01 2.61748064248073760221235950496066e-03 2.37834454754113411301119640484103e-01 1.56072094208308561995302010672049e-03
            6.34771577976172451940328755881637e-01 3.97575641532846215525642818988672e-03 3.61252665608499112792628693568986e-01 2.67163721914266986609542975372733e-03
            5.00000000000000111022302462515654e-01 5.44283546348575588552654025420452e-03 4.94557164536514159980384874870651e-01 3.79833712774578762666477516063424e-03
            3.65228422023827603570822475376190e-01 6.90991451164305135052012829532941e-03 6.27861663464529429212745981203625e-01 4.64333899310942575106020413500119e-03
            2.40451935396594040916795620432822e-01 8.26819028449077633724506597445725e-03 7.51279874318915186215406265546335e-01 4.93006042554818176004705065906819e-03
            1.34923997212975321779993009840837e-01 9.41693269315944242459082147433946e-03 8.55659070093865214978734456963139e-01 4.48563928392311551951454617892523e-03
            5.64687001159524526450184112036368e-02 1.02709712398354156837054773632190e-02 9.33260328644212044935102312592790e-01 3.29805668173470010670622620807535e-03
            1.08856709269715135057765564852161e-02 1.07671730954411997566166903084195e-02 9.78347155977587257247307661600644e-01 1.53262739316846192776100377130888e-03
            ];
        
    case  21
        
        % ALG. DEG.:   21
        % PTS CARD.:  144
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.3e-15
        
        xyw_bar=[
            9.90780317123359566267026821151376e-01 9.13468032429461639087886482002432e-03 8.50025523458173420943140285999107e-05 1.02592591132837378156000829321215e-05
            9.52058628185237454388811784156132e-01 4.74993675699593789762964490819286e-02 4.42004244803166634891766761938925e-04 1.20929681615561241650631807420524e-04
            8.84951337097152235244834628247190e-01 1.13987950715501940024410032492597e-01 1.06071218734582473075533926021308e-03 4.34409312354210985340080553029907e-04
            7.93658977143308685953115855227225e-01 2.04438624061510948459741143778956e-01 1.90239879518036558714300099381944e-03 9.88836910309064337845641468049962e-04
            6.83915749499090064311701553378953e-01 3.13170053948990945169583710594452e-01 2.91419655191899051871473602659535e-03 1.74084806826103064350541593796606e-03
            5.62616704255734512152287152275676e-01 4.33350760461963468994639470110997e-01 4.03253528230201885307337761332747e-03 2.57041321117076936378365026314441e-03
            4.37383295744265598870015310239978e-01 5.57429556661395930738933657266898e-01 5.18714759433847039105103249312378e-03 3.30638463680572912764543680452789e-03
            3.16084250500909769154844752847566e-01 6.77610263174368676608594341814751e-01 6.30548632472149872540967407985590e-03 3.76669640920740660811061317758686e-03
            2.06341022856691147513430450999294e-01 7.86341693061848756585163755516987e-01 7.31728408146015141255702474154532e-03 3.80340893891226161649399273301242e-03
            1.15048662902847709244014140494983e-01 8.76792366407857515220314326143125e-01 8.15897068929472002452030210406519e-03 3.34146518625726010945364841120409e-03
            4.79413718147626011223394471016945e-02 9.43280949553400249740775507234503e-01 8.77767863183720464803627692162991e-03 2.40151965677245669492090307528542e-03
            9.21968287664043373297317884862423e-03 9.81645636799064913446954960818402e-01 9.13468032429465282007186033297330e-03 1.10249692247699334832211626178378e-03
            9.90780317123359566267026821151376e-01 8.77767863183721505637713278247247e-03 4.42004244803218676596046066151757e-04 2.32561829723288764841093029245300e-05
            9.52058628185237454388811784156132e-01 4.56429966832812908372751792285271e-02 2.29837513148125477391303661534039e-03 2.74129230130810042261046977074557e-04
            8.84951337097152235244834628247190e-01 1.09533072177831009019755015287956e-01 5.51559072501675573541035646485398e-03 9.84739964303277086343779700428058e-04
            7.93658977143308685953115855227225e-01 1.96448751159280204925394741621858e-01 9.89227169741110912148940315091750e-03 2.24154316232873847963835700625168e-03
            6.83915749499090064311701553378953e-01 3.00930737922855240373820606691879e-01 1.51535125780546953144778399291681e-02 3.94623829610493085523481582299610e-03
            5.62616704255734512152287152275676e-01 4.16414540537423327748456358676776e-01 2.09687552068421600992564890475478e-02 5.82673654046596788125267352143055e-03
            4.37383295744265598870015310239978e-01 5.35644087647813904595750500448048e-01 2.69726166079204965342341893119738e-02 7.49507203603899923444187081145174e-03
            3.16084250500909769154844752847566e-01 6.51127890262382269526142408722080e-01 3.27878592367079058078616071725264e-02 8.53852894506953660758163948685251e-03
            2.06341022856691147513430450999294e-01 7.55609877025957388241295120678842e-01 3.80491001173515197564256595796905e-02 8.62175067665517033677069491659495e-03
            1.15048662902847709244014140494983e-01 8.42525556007406417613481153239263e-01 4.24257810897458176313534750079270e-02 7.57459431613798691435013310524482e-03
            4.79413718147626011223394471016945e-02 9.06415631501956253757157355721574e-01 4.56429966832812006316544284345582e-02 5.44388049203568900186178680655757e-03
            9.21968287664043373297317884862423e-03 9.43280949553400249740775507234503e-01 4.74993675699593165262513139168732e-02 2.49919315541565932245471337580511e-03
            9.90780317123359566267026821151376e-01 8.15897068929467145226297475346655e-03 1.06071218734576228071020409515768e-03 3.48123654591915259576793972851050e-05
            9.52058628185237454388811784156132e-01 4.24257810897458939591864179874392e-02 5.51559072501665165200179785642831e-03 4.10346227225478793650126352332563e-04
            8.84951337097152235244834628247190e-01 1.01812468067114622605195961568825e-01 1.32361948357331421499694101839850e-02 1.47406509315763187967229796981883e-03
            7.93658977143308685953115855227225e-01 1.82601764075022987565688481481629e-01 2.37392587816683264811956632911460e-02 3.35538380706701345676523651206935e-03
            6.83915749499090064311701553378953e-01 2.79719180116131449498340089121484e-01 3.63650703847784861899583574995631e-02 5.90715552575931009127163662242310e-03
            5.62616704255734512152287152275676e-01 3.87062932392846892337701092401403e-01 5.03203633514185955100117553229211e-02 8.72208832044693571639015772234416e-03
            4.37383295744265598870015310239978e-01 4.97888404704305231884831073330133e-01 6.47282995514291692451536164298886e-02 1.12194330072140670273572737869472e-02
            3.16084250500909769154844752847566e-01 6.05232156981020841257645770383533e-01 7.86835925180693340763582455110736e-02 1.27813919624437614258605222516962e-02
            2.06341022856691147513430450999294e-01 7.02349573022129414212599840539042e-01 9.13094041211794937851209397194907e-02 1.29059672350735378520969476312530e-02
            1.15048662902847709244014140494983e-01 7.83138869030037598761850858863909e-01 1.01812468067114636482983769383281e-01 1.13384705414581021115649761554778e-02
            4.79413718147626011223394471016945e-02 8.42525556007406417613481153239263e-01 1.09533072177831036775330630916869e-01 8.14898805321584603078566289013906e-03
            9.21968287664043373297317884862423e-03 8.76792366407857515220314326143125e-01 1.13987950715502051046712495008251e-01 3.74106213315226207929975466015549e-03
            9.90780317123359566267026821151376e-01 7.31728408146012452434314710103536e-03 1.90239879518030920863003174758887e-03 4.41829869967171829574052732692024e-05
            9.52058628185237454388811784156132e-01 3.80491001173515336342134673941473e-02 9.89227169741101197697474844972021e-03 5.20801209067748779868445296870050e-04
            8.84951337097152235244834628247190e-01 9.13094041211794382739697084616637e-02 2.37392587816683264811956632911460e-02 1.87084669439210480798574387506505e-03
            7.93658977143308685953115855227225e-01 1.63764405143145663767967334933928e-01 4.25766177135456502789168098388473e-02 4.25856953875837496159562078901217e-03
            6.83915749499090064311701553378953e-01 2.50863102943661542276743148249807e-01 6.52211475572483934115552983712405e-02 7.49721463449990344757312143997297e-03
            5.62616704255734512152287152275676e-01 3.47133179119962997916104541218374e-01 9.02501166243024899316083065059502e-02 1.10698572120377664723678279301566e-02
            4.37383295744265598870015310239978e-01 4.46525798023345577014708851493197e-01 1.16090906232388824115275838266825e-01 1.42394248747436565072366221329503e-02
            3.16084250500909769154844752847566e-01 5.42795874199647254698675169493072e-01 1.41119875299442920635328846401535e-01 1.62218242692696683338038354804667e-02
            2.06341022856691147513430450999294e-01 6.29894572000163188718602214066777e-01 1.63764405143145719279118566191755e-01 1.63799321018777699343615950056119e-02
            1.15048662902847709244014140494983e-01 7.02349573022129192167994915507734e-01 1.82601764075023043076839712739456e-01 1.43905043477484655828346760131353e-02
            4.79413718147626011223394471016945e-02 7.55609877025957166196690195647534e-01 1.96448751159280288192121588508599e-01 1.03424926299162493747996194315419e-02
            9.21968287664043373297317884862423e-03 7.86341693061848534540558830485679e-01 2.04438624061511031726467990665697e-01 4.74806286222459201368240044871527e-03
            9.90780317123359566267026821151376e-01 6.30548632472146923511058247413530e-03 2.91419655191896449786259637448893e-03 5.07778135112511595024471777737318e-05
            9.52058628185237454388811784156132e-01 3.27878592367079127467555110797548e-02 1.51535125780546328644327047641127e-02 5.98536868329910894288847789113106e-04
            8.84951337097152235244834628247190e-01 7.86835925180692508096313986243331e-02 3.63650703847785139455339731284766e-02 2.15009240011412904383658606377594e-03
            7.93658977143308685953115855227225e-01 1.41119875299442865124177615143708e-01 6.52211475572484489227065296290675e-02 4.89421074858144983243946057882567e-03
            6.83915749499090064311701553378953e-01 2.16174997086187953287961249770888e-01 9.99092534147219824003371968501597e-02 8.61626142643422479605597175122966e-03
            5.62616704255734512152287152275676e-01 2.99133324527321442509730786696309e-01 1.38249971216944045337982061028015e-01 1.27221626086710028896709090417971e-02
            4.37383295744265598870015310239978e-01 3.84782424971768510779668304166989e-01 1.77834279283965890350316385593032e-01 1.63648252403335026095021476066904e-02
            3.16084250500909769154844752847566e-01 4.67740752412902194290467150494806e-01 2.16174997086187981043536865399801e-01 1.86431208831233027889417996902921e-02
            2.06341022856691147513430450999294e-01 5.42795874199647254698675169493072e-01 2.50863102943661653299045610765461e-01 1.88248281551879718953568243478003e-02
            1.15048662902847709244014140494983e-01 6.05232156981020730235343307867879e-01 2.79719180116131505009491320379311e-01 1.65384550880888457624617871033479e-02
            4.79413718147626011223394471016945e-02 6.51127890262382158503839946206426e-01 3.00930737922855295884971837949706e-01 1.18862303728445809630764884445853e-02
            9.21968287664043373297317884862423e-03 6.77610263174368565586291879299097e-01 3.13170053948991000680734941852279e-01 5.45676666395713150931046087066534e-03
            9.90780317123359566267026821151376e-01 5.18714759433847039105103249312378e-03 4.03253528230196334192214635550044e-03 5.41822124884663202304055373836889e-05
            9.52058628185237454388811784156132e-01 2.69726166079204965342341893119738e-02 2.09687552068420490769540265318938e-02 6.38665778211318369006477269067545e-04
            8.84951337097152235244834628247190e-01 6.47282995514291692451536164298886e-02 5.03203633514185955100117553229211e-02 2.29424536499602026509703200929380e-03
            7.93658977143308685953115855227225e-01 1.16090906232388824115275838266825e-01 9.02501166243024899316083065059502e-02 5.22234315355501674260629840773618e-03
            6.83915749499090064311701553378953e-01 1.77834279283965945861467616850859e-01 1.38249971216943989826830829770188e-01 9.19393875358167148026389270398795e-03
            5.62616704255734512152287152275676e-01 2.46079148348149839842236019649135e-01 1.91304147396115648005476828075189e-01 1.35751201186143505283965282615100e-02
            4.37383295744265598870015310239978e-01 3.16537555907584589043324285739800e-01 2.46079148348149812086660404020222e-01 1.74620051001585369077062637188646e-02
            3.16084250500909769154844752847566e-01 3.84782424971768621801970766682643e-01 2.99133324527321553532033249211963e-01 1.98930490954229571842759582978033e-02
            2.06341022856691147513430450999294e-01 4.46525798023345799059313776524505e-01 3.47133179119963108938407003734028e-01 2.00869389332263524605703963743508e-02
            1.15048662902847709244014140494983e-01 4.97888404704305342907133535845787e-01 3.87062932392846892337701092401403e-01 1.76472759626649620956495567725142e-02
            4.79413718147626011223394471016945e-02 5.35644087647814015618052962963702e-01 4.16414540537423438770758821192430e-01 1.26831427982937311460531759621517e-02
            9.21968287664043373297317884862423e-03 5.57429556661396041761236119782552e-01 4.33350760461963524505790701368824e-01 5.82261563548799963158097980908678e-03
            9.90780317123359566267026821151376e-01 4.03253528230196420928388434390399e-03 5.18714759433846952368929450472024e-03 5.41822124884662660202969131084672e-05
            9.52058628185237454388811784156132e-01 2.09687552068421115269991616969492e-02 2.69726166079204340841890541469184e-02 6.38665778211317718485173777764885e-04
            8.84951337097152235244834628247190e-01 5.03203633514185608155422357867792e-02 6.47282995514291970007292320588022e-02 2.29424536499601809669268703828493e-03
            7.93658977143308685953115855227225e-01 9.02501166243024621760326908770367e-02 1.16090906232388851870851453895739e-01 5.22234315355501153843587047731489e-03
            6.83915749499090064311701553378953e-01 1.38249971216944017582406445399101e-01 1.77834279283965918105892001221946e-01 9.19393875358166107192303684314538e-03
            5.62616704255734512152287152275676e-01 1.91304147396115620249901212446275e-01 2.46079148348149867597811635278049e-01 1.35751201186143366506087204470532e-02
            4.37383295744265598870015310239978e-01 2.46079148348149895353387250906962e-01 3.16537555907584478021021823224146e-01 1.74620051001585195604715039507937e-02
            3.16084250500909769154844752847566e-01 2.99133324527321609043184480469790e-01 3.84782424971768566290819535424816e-01 1.98930490954229363675942465761182e-02
            2.06341022856691147513430450999294e-01 3.47133179119963219960709466249682e-01 4.46525798023345688037011314008851e-01 2.00869389332263351133356366062799e-02
            1.15048662902847709244014140494983e-01 3.87062932392847003360003554917057e-01 4.97888404704305231884831073330133e-01 1.76472759626649447484147970044432e-02
            4.79413718147626011223394471016945e-02 4.16414540537423494281910052450257e-01 5.35644087647813904595750500448048e-01 1.26831427982937190029888441245021e-02
            9.21968287664043373297317884862423e-03 4.33350760461963635528093163884478e-01 5.57429556661395930738933657266898e-01 5.82261563548799356004881389026195e-03
            9.90780317123359566267026821151376e-01 2.91419655191896319681998939188361e-03 6.30548632472147096983405845094239e-03 5.07778135112514102241995650466322e-05
            9.52058628185237454388811784156132e-01 1.51535125780546779672430801610972e-02 3.27878592367078641744981837291562e-02 5.98536868329913821634713499975078e-04
            8.84951337097152235244834628247190e-01 3.63650703847784376177010301489645e-02 7.86835925180693340763582455110736e-02 2.15009240011413945217744192461851e-03
            7.93658977143308685953115855227225e-01 6.52211475572483517781918749278702e-02 1.41119875299442976146480077659362e-01 4.89421074858147325120638626572145e-03
            6.83915749499090064311701553378953e-01 9.99092534147219268891859655923326e-02 2.16174997086188008799112481028715e-01 8.61626142643426642941939519459993e-03
            5.62616704255734512152287152275676e-01 1.38249971216943906560103982883447e-01 2.99133324527321553532033249211963e-01 1.27221626086710636049925682300454e-02
            4.37383295744265598870015310239978e-01 1.77834279283965807083589538706292e-01 3.84782424971768621801970766682643e-01 1.63648252403335824067820425398168e-02
            3.16084250500909769154844752847566e-01 2.16174997086187870021234402884147e-01 4.67740752412902305312769613010460e-01 1.86431208831233929945625504842610e-02
            2.06341022856691147513430450999294e-01 2.50863102943661486765591916991980e-01 5.42795874199647476743280094524380e-01 1.88248281551880655704245270953834e-02
            1.15048662902847709244014140494983e-01 2.79719180116131338476037626605830e-01 6.05232156981020841257645770383533e-01 1.65384550880889255597416820364742e-02
            4.79413718147626011223394471016945e-02 3.00930737922855129351518144176225e-01 6.51127890262382269526142408722080e-01 1.18862303728446382089511956792194e-02
            9.21968287664043373297317884862423e-03 3.13170053948990778636130016820971e-01 6.77610263174368787630896804330405e-01 5.45676666395715839752433851117530e-03
            9.90780317123359566267026821151376e-01 1.90239879518030769074699026788267e-03 7.31728408146012625906662307784245e-03 4.41829869967170135508158224091346e-05
            9.52058628185237454388811784156132e-01 9.89227169741105881450859982351176e-03 3.80491001173514850619561400435487e-02 5.20801209067746828304534822962069e-04
            8.84951337097152235244834628247190e-01 2.37392587816682536228096722652481e-02 9.13094041211795076629087475339475e-02 1.87084669439209765225140547073579e-03
            7.93658977143308685953115855227225e-01 4.25766177135455600732960590448783e-02 1.63764405143145747034694181820669e-01 4.25856953875835848172259900934478e-03
            6.83915749499090064311701553378953e-01 6.52211475572483517781918749278702e-02 2.50863102943661597787894379507634e-01 7.49721463449987482463576782265591e-03
            5.62616704255734512152287152275676e-01 9.02501166243023789093058439902961e-02 3.47133179119963108938407003734028e-01 1.10698572120377248390044044867864e-02
            4.37383295744265598870015310239978e-01 1.16090906232388740848548991380085e-01 4.46525798023345688037011314008851e-01 1.42394248747436009960853908751233e-02
            3.16084250500909769154844752847566e-01 1.41119875299442809613026383885881e-01 5.42795874199647365720977632008726e-01 1.62218242692696058837587003154113e-02
            2.06341022856691147513430450999294e-01 1.63764405143145608256816103676101e-01 6.29894572000163299740904676582431e-01 1.63799321018777074843164598405565e-02
            1.15048662902847709244014140494983e-01 1.82601764075022876543386018965975e-01 7.02349573022129414212599840539042e-01 1.43905043477484100716834447553083e-02
            4.79413718147626011223394471016945e-02 1.96448751159280093903092279106204e-01 7.55609877025957388241295120678842e-01 1.03424926299162094761596719649788e-02
            9.21968287664043373297317884862423e-03 2.04438624061510837437438681263302e-01 7.86341693061848756585163755516987e-01 4.74806286222457379908590269224078e-03
            9.90780317123359566267026821151376e-01 1.06071218734576249755063859225857e-03 8.15897068929467145226297475346655e-03 3.48123654591914581950436169410779e-05
            9.52058628185237454388811784156132e-01 5.51559072501670716315302911425533e-03 4.24257810897458384480351867296122e-02 4.10346227225478034708605612479460e-04
            8.84951337097152235244834628247190e-01 1.32361948357330866388181789261580e-02 1.01812468067114678116347192826652e-01 1.47406509315762906074664950750730e-03
            7.93658977143308685953115855227225e-01 2.37392587816682675005974800797048e-02 1.82601764075023043076839712739456e-01 3.35538380706700695155220159904275e-03
            6.83915749499090064311701553378953e-01 3.63650703847784584343827418706496e-02 2.79719180116131505009491320379311e-01 5.90715552575929881556904277317699e-03
            5.62616704255734512152287152275676e-01 5.03203633514185399988605240650941e-02 3.87062932392846947848852323659230e-01 8.72208832044691836915539795427321e-03
            4.37383295744265598870015310239978e-01 6.47282995514291553673658086154319e-02 4.97888404704305231884831073330133e-01 1.12194330072140462106755620652621e-02
            3.16084250500909769154844752847566e-01 7.86835925180692646874192064387898e-02 6.05232156981020952279948232899187e-01 1.27813919624437371397318585763969e-02
            2.06341022856691147513430450999294e-01 9.13094041211794660295453240905772e-02 7.02349573022129414212599840539042e-01 1.29059672350735135659682839559537e-02
            1.15048662902847709244014140494983e-01 1.01812468067114622605195961568825e-01 7.83138869030037598761850858863909e-01 1.13384705414580795601597884569856e-02
            4.79413718147626011223394471016945e-02 1.09533072177831009019755015287956e-01 8.42525556007406417613481153239263e-01 8.14898805321583041827437909887522e-03
            9.21968287664043373297317884862423e-03 1.13987950715501940024410032492597e-01 8.76792366407857626242616788658779e-01 3.74106213315225470672498175872533e-03
            9.90780317123359566267026821151376e-01 4.42004244803219056066806436078309e-04 8.77767863183721505637713278247247e-03 2.32561829723290052331172855781816e-05
            9.52058628185237454388811784156132e-01 2.29837513148131375451121982678160e-03 4.56429966832812283872300440634717e-02 2.74129230130811560144088456780764e-04
            8.84951337097152235244834628247190e-01 5.51559072501670716315302911425533e-03 1.09533072177831064530906246545783e-01 9.84739964303282724195076625051115e-04
            7.93658977143308685953115855227225e-01 9.89227169741106401867902775393304e-03 1.96448751159280260436545972879685e-01 2.24154316232875105638355783810312e-03
            6.83915749499090064311701553378953e-01 1.51535125780546849061369840683255e-02 3.00930737922855240373820606691879e-01 3.94623829610495253927826553308478e-03
            5.62616704255734512152287152275676e-01 2.09687552068421011186583058361066e-02 4.16414540537423383259607589934603e-01 5.82673654046600084099871708076535e-03
            4.37383295744265598870015310239978e-01 2.69726166079204895953402854047454e-02 5.35644087647813904595750500448048e-01 7.49507203603904086780529425482200e-03
            3.16084250500909769154844752847566e-01 3.27878592367079196856494149869832e-02 6.51127890262382269526142408722080e-01 8.53852894506958344511549086064406e-03
            2.06341022856691147513430450999294e-01 3.80491001173515475120012752086041e-02 7.55609877025957388241295120678842e-01 8.62175067665521890902802226719359e-03
            1.15048662902847709244014140494983e-01 4.24257810897458939591864179874392e-02 8.42525556007406306591178690723609e-01 7.57459431613802854771355654861509e-03
            4.79413718147626011223394471016945e-02 4.56429966832812908372751792285271e-02 9.06415631501956142734854893205920e-01 5.44388049203571935952261640068173e-03
            9.21968287664043373297317884862423e-03 4.74993675699593789762964490819286e-02 9.43280949553400138718473044718849e-01 2.49919315541567363392339018446364e-03
            9.90780317123359566267026821151376e-01 8.50025523458168270982820979853045e-05 9.13468032429461639087886482002432e-03 1.02592591132836700529643025880944e-05
            9.52058628185237454388811784156132e-01 4.42004244803219056066806436078309e-04 4.74993675699593234651452178241016e-02 1.20929681615560455604056755429809e-04
            8.84951337097152235244834628247190e-01 1.06071218734576249755063859225857e-03 1.13987950715501995535561263750424e-01 4.34409312354208166414432090718378e-04
            7.93658977143308685953115855227225e-01 1.90239879518030877494916275338710e-03 2.04438624061511003970892375036783e-01 9.88836910309057832632606555023358e-04
            6.83915749499090064311701553378953e-01 2.91419655191896493154346536869070e-03 3.13170053948990945169583710594452e-01 1.74084806826101936780282208871995e-03
            5.62616704255734512152287152275676e-01 4.03253528230196247456040836709690e-03 4.33350760461963524505790701368824e-01 2.57041321117075288391062848347701e-03
            4.37383295744265598870015310239978e-01 5.18714759433846952368929450472024e-03 5.57429556661395930738933657266898e-01 3.30638463680570787728285608864098e-03
            3.16084250500909769154844752847566e-01 6.30548632472147010247232046253885e-03 6.77610263174368676608594341814751e-01 3.76669640920738232198194950228753e-03
            2.06341022856691147513430450999294e-01 7.31728408146012625906662307784245e-03 7.86341693061848756585163755516987e-01 3.80340893891223689668446006351132e-03
            1.15048662902847709244014140494983e-01 8.15897068929467145226297475346655e-03 8.76792366407857515220314326143125e-01 3.34146518625723842541019870111541e-03
            4.79413718147626011223394471016945e-02 8.77767863183721505637713278247247e-03 9.43280949553400249740775507234503e-01 2.40151965677244151609048827822335e-03
            9.21968287664043373297317884862423e-03 9.13468032429461639087886482002432e-03 9.81645636799064913446954960818402e-01 1.10249692247698619258777785745451e-03
            ];
        
    case  22
        
        % ALG. DEG.:   22
        % PTS CARD.:  144
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.3e-15
        
        xyw_bar=[
            9.90780317123359566267026821151376e-01 9.13468032429461639087886482002432e-03 8.50025523458173420943140285999107e-05 1.02592591132837378156000829321215e-05
            9.52058628185237454388811784156132e-01 4.74993675699593789762964490819286e-02 4.42004244803166634891766761938925e-04 1.20929681615561241650631807420524e-04
            8.84951337097152235244834628247190e-01 1.13987950715501940024410032492597e-01 1.06071218734582473075533926021308e-03 4.34409312354210985340080553029907e-04
            7.93658977143308685953115855227225e-01 2.04438624061510948459741143778956e-01 1.90239879518036558714300099381944e-03 9.88836910309064337845641468049962e-04
            6.83915749499090064311701553378953e-01 3.13170053948990945169583710594452e-01 2.91419655191899051871473602659535e-03 1.74084806826103064350541593796606e-03
            5.62616704255734512152287152275676e-01 4.33350760461963468994639470110997e-01 4.03253528230201885307337761332747e-03 2.57041321117076936378365026314441e-03
            4.37383295744265598870015310239978e-01 5.57429556661395930738933657266898e-01 5.18714759433847039105103249312378e-03 3.30638463680572912764543680452789e-03
            3.16084250500909769154844752847566e-01 6.77610263174368676608594341814751e-01 6.30548632472149872540967407985590e-03 3.76669640920740660811061317758686e-03
            2.06341022856691147513430450999294e-01 7.86341693061848756585163755516987e-01 7.31728408146015141255702474154532e-03 3.80340893891226161649399273301242e-03
            1.15048662902847709244014140494983e-01 8.76792366407857515220314326143125e-01 8.15897068929472002452030210406519e-03 3.34146518625726010945364841120409e-03
            4.79413718147626011223394471016945e-02 9.43280949553400249740775507234503e-01 8.77767863183720464803627692162991e-03 2.40151965677245669492090307528542e-03
            9.21968287664043373297317884862423e-03 9.81645636799064913446954960818402e-01 9.13468032429465282007186033297330e-03 1.10249692247699334832211626178378e-03
            9.90780317123359566267026821151376e-01 8.77767863183721505637713278247247e-03 4.42004244803218676596046066151757e-04 2.32561829723288764841093029245300e-05
            9.52058628185237454388811784156132e-01 4.56429966832812908372751792285271e-02 2.29837513148125477391303661534039e-03 2.74129230130810042261046977074557e-04
            8.84951337097152235244834628247190e-01 1.09533072177831009019755015287956e-01 5.51559072501675573541035646485398e-03 9.84739964303277086343779700428058e-04
            7.93658977143308685953115855227225e-01 1.96448751159280204925394741621858e-01 9.89227169741110912148940315091750e-03 2.24154316232873847963835700625168e-03
            6.83915749499090064311701553378953e-01 3.00930737922855240373820606691879e-01 1.51535125780546953144778399291681e-02 3.94623829610493085523481582299610e-03
            5.62616704255734512152287152275676e-01 4.16414540537423327748456358676776e-01 2.09687552068421600992564890475478e-02 5.82673654046596788125267352143055e-03
            4.37383295744265598870015310239978e-01 5.35644087647813904595750500448048e-01 2.69726166079204965342341893119738e-02 7.49507203603899923444187081145174e-03
            3.16084250500909769154844752847566e-01 6.51127890262382269526142408722080e-01 3.27878592367079058078616071725264e-02 8.53852894506953660758163948685251e-03
            2.06341022856691147513430450999294e-01 7.55609877025957388241295120678842e-01 3.80491001173515197564256595796905e-02 8.62175067665517033677069491659495e-03
            1.15048662902847709244014140494983e-01 8.42525556007406417613481153239263e-01 4.24257810897458176313534750079270e-02 7.57459431613798691435013310524482e-03
            4.79413718147626011223394471016945e-02 9.06415631501956253757157355721574e-01 4.56429966832812006316544284345582e-02 5.44388049203568900186178680655757e-03
            9.21968287664043373297317884862423e-03 9.43280949553400249740775507234503e-01 4.74993675699593165262513139168732e-02 2.49919315541565932245471337580511e-03
            9.90780317123359566267026821151376e-01 8.15897068929467145226297475346655e-03 1.06071218734576228071020409515768e-03 3.48123654591915259576793972851050e-05
            9.52058628185237454388811784156132e-01 4.24257810897458939591864179874392e-02 5.51559072501665165200179785642831e-03 4.10346227225478793650126352332563e-04
            8.84951337097152235244834628247190e-01 1.01812468067114622605195961568825e-01 1.32361948357331421499694101839850e-02 1.47406509315763187967229796981883e-03
            7.93658977143308685953115855227225e-01 1.82601764075022987565688481481629e-01 2.37392587816683264811956632911460e-02 3.35538380706701345676523651206935e-03
            6.83915749499090064311701553378953e-01 2.79719180116131449498340089121484e-01 3.63650703847784861899583574995631e-02 5.90715552575931009127163662242310e-03
            5.62616704255734512152287152275676e-01 3.87062932392846892337701092401403e-01 5.03203633514185955100117553229211e-02 8.72208832044693571639015772234416e-03
            4.37383295744265598870015310239978e-01 4.97888404704305231884831073330133e-01 6.47282995514291692451536164298886e-02 1.12194330072140670273572737869472e-02
            3.16084250500909769154844752847566e-01 6.05232156981020841257645770383533e-01 7.86835925180693340763582455110736e-02 1.27813919624437614258605222516962e-02
            2.06341022856691147513430450999294e-01 7.02349573022129414212599840539042e-01 9.13094041211794937851209397194907e-02 1.29059672350735378520969476312530e-02
            1.15048662902847709244014140494983e-01 7.83138869030037598761850858863909e-01 1.01812468067114636482983769383281e-01 1.13384705414581021115649761554778e-02
            4.79413718147626011223394471016945e-02 8.42525556007406417613481153239263e-01 1.09533072177831036775330630916869e-01 8.14898805321584603078566289013906e-03
            9.21968287664043373297317884862423e-03 8.76792366407857515220314326143125e-01 1.13987950715502051046712495008251e-01 3.74106213315226207929975466015549e-03
            9.90780317123359566267026821151376e-01 7.31728408146012452434314710103536e-03 1.90239879518030920863003174758887e-03 4.41829869967171829574052732692024e-05
            9.52058628185237454388811784156132e-01 3.80491001173515336342134673941473e-02 9.89227169741101197697474844972021e-03 5.20801209067748779868445296870050e-04
            8.84951337097152235244834628247190e-01 9.13094041211794382739697084616637e-02 2.37392587816683264811956632911460e-02 1.87084669439210480798574387506505e-03
            7.93658977143308685953115855227225e-01 1.63764405143145663767967334933928e-01 4.25766177135456502789168098388473e-02 4.25856953875837496159562078901217e-03
            6.83915749499090064311701553378953e-01 2.50863102943661542276743148249807e-01 6.52211475572483934115552983712405e-02 7.49721463449990344757312143997297e-03
            5.62616704255734512152287152275676e-01 3.47133179119962997916104541218374e-01 9.02501166243024899316083065059502e-02 1.10698572120377664723678279301566e-02
            4.37383295744265598870015310239978e-01 4.46525798023345577014708851493197e-01 1.16090906232388824115275838266825e-01 1.42394248747436565072366221329503e-02
            3.16084250500909769154844752847566e-01 5.42795874199647254698675169493072e-01 1.41119875299442920635328846401535e-01 1.62218242692696683338038354804667e-02
            2.06341022856691147513430450999294e-01 6.29894572000163188718602214066777e-01 1.63764405143145719279118566191755e-01 1.63799321018777699343615950056119e-02
            1.15048662902847709244014140494983e-01 7.02349573022129192167994915507734e-01 1.82601764075023043076839712739456e-01 1.43905043477484655828346760131353e-02
            4.79413718147626011223394471016945e-02 7.55609877025957166196690195647534e-01 1.96448751159280288192121588508599e-01 1.03424926299162493747996194315419e-02
            9.21968287664043373297317884862423e-03 7.86341693061848534540558830485679e-01 2.04438624061511031726467990665697e-01 4.74806286222459201368240044871527e-03
            9.90780317123359566267026821151376e-01 6.30548632472146923511058247413530e-03 2.91419655191896449786259637448893e-03 5.07778135112511595024471777737318e-05
            9.52058628185237454388811784156132e-01 3.27878592367079127467555110797548e-02 1.51535125780546328644327047641127e-02 5.98536868329910894288847789113106e-04
            8.84951337097152235244834628247190e-01 7.86835925180692508096313986243331e-02 3.63650703847785139455339731284766e-02 2.15009240011412904383658606377594e-03
            7.93658977143308685953115855227225e-01 1.41119875299442865124177615143708e-01 6.52211475572484489227065296290675e-02 4.89421074858144983243946057882567e-03
            6.83915749499090064311701553378953e-01 2.16174997086187953287961249770888e-01 9.99092534147219824003371968501597e-02 8.61626142643422479605597175122966e-03
            5.62616704255734512152287152275676e-01 2.99133324527321442509730786696309e-01 1.38249971216944045337982061028015e-01 1.27221626086710028896709090417971e-02
            4.37383295744265598870015310239978e-01 3.84782424971768510779668304166989e-01 1.77834279283965890350316385593032e-01 1.63648252403335026095021476066904e-02
            3.16084250500909769154844752847566e-01 4.67740752412902194290467150494806e-01 2.16174997086187981043536865399801e-01 1.86431208831233027889417996902921e-02
            2.06341022856691147513430450999294e-01 5.42795874199647254698675169493072e-01 2.50863102943661653299045610765461e-01 1.88248281551879718953568243478003e-02
            1.15048662902847709244014140494983e-01 6.05232156981020730235343307867879e-01 2.79719180116131505009491320379311e-01 1.65384550880888457624617871033479e-02
            4.79413718147626011223394471016945e-02 6.51127890262382158503839946206426e-01 3.00930737922855295884971837949706e-01 1.18862303728445809630764884445853e-02
            9.21968287664043373297317884862423e-03 6.77610263174368565586291879299097e-01 3.13170053948991000680734941852279e-01 5.45676666395713150931046087066534e-03
            9.90780317123359566267026821151376e-01 5.18714759433847039105103249312378e-03 4.03253528230196334192214635550044e-03 5.41822124884663202304055373836889e-05
            9.52058628185237454388811784156132e-01 2.69726166079204965342341893119738e-02 2.09687552068420490769540265318938e-02 6.38665778211318369006477269067545e-04
            8.84951337097152235244834628247190e-01 6.47282995514291692451536164298886e-02 5.03203633514185955100117553229211e-02 2.29424536499602026509703200929380e-03
            7.93658977143308685953115855227225e-01 1.16090906232388824115275838266825e-01 9.02501166243024899316083065059502e-02 5.22234315355501674260629840773618e-03
            6.83915749499090064311701553378953e-01 1.77834279283965945861467616850859e-01 1.38249971216943989826830829770188e-01 9.19393875358167148026389270398795e-03
            5.62616704255734512152287152275676e-01 2.46079148348149839842236019649135e-01 1.91304147396115648005476828075189e-01 1.35751201186143505283965282615100e-02
            4.37383295744265598870015310239978e-01 3.16537555907584589043324285739800e-01 2.46079148348149812086660404020222e-01 1.74620051001585369077062637188646e-02
            3.16084250500909769154844752847566e-01 3.84782424971768621801970766682643e-01 2.99133324527321553532033249211963e-01 1.98930490954229571842759582978033e-02
            2.06341022856691147513430450999294e-01 4.46525798023345799059313776524505e-01 3.47133179119963108938407003734028e-01 2.00869389332263524605703963743508e-02
            1.15048662902847709244014140494983e-01 4.97888404704305342907133535845787e-01 3.87062932392846892337701092401403e-01 1.76472759626649620956495567725142e-02
            4.79413718147626011223394471016945e-02 5.35644087647814015618052962963702e-01 4.16414540537423438770758821192430e-01 1.26831427982937311460531759621517e-02
            9.21968287664043373297317884862423e-03 5.57429556661396041761236119782552e-01 4.33350760461963524505790701368824e-01 5.82261563548799963158097980908678e-03
            9.90780317123359566267026821151376e-01 4.03253528230196420928388434390399e-03 5.18714759433846952368929450472024e-03 5.41822124884662660202969131084672e-05
            9.52058628185237454388811784156132e-01 2.09687552068421115269991616969492e-02 2.69726166079204340841890541469184e-02 6.38665778211317718485173777764885e-04
            8.84951337097152235244834628247190e-01 5.03203633514185608155422357867792e-02 6.47282995514291970007292320588022e-02 2.29424536499601809669268703828493e-03
            7.93658977143308685953115855227225e-01 9.02501166243024621760326908770367e-02 1.16090906232388851870851453895739e-01 5.22234315355501153843587047731489e-03
            6.83915749499090064311701553378953e-01 1.38249971216944017582406445399101e-01 1.77834279283965918105892001221946e-01 9.19393875358166107192303684314538e-03
            5.62616704255734512152287152275676e-01 1.91304147396115620249901212446275e-01 2.46079148348149867597811635278049e-01 1.35751201186143366506087204470532e-02
            4.37383295744265598870015310239978e-01 2.46079148348149895353387250906962e-01 3.16537555907584478021021823224146e-01 1.74620051001585195604715039507937e-02
            3.16084250500909769154844752847566e-01 2.99133324527321609043184480469790e-01 3.84782424971768566290819535424816e-01 1.98930490954229363675942465761182e-02
            2.06341022856691147513430450999294e-01 3.47133179119963219960709466249682e-01 4.46525798023345688037011314008851e-01 2.00869389332263351133356366062799e-02
            1.15048662902847709244014140494983e-01 3.87062932392847003360003554917057e-01 4.97888404704305231884831073330133e-01 1.76472759626649447484147970044432e-02
            4.79413718147626011223394471016945e-02 4.16414540537423494281910052450257e-01 5.35644087647813904595750500448048e-01 1.26831427982937190029888441245021e-02
            9.21968287664043373297317884862423e-03 4.33350760461963635528093163884478e-01 5.57429556661395930738933657266898e-01 5.82261563548799356004881389026195e-03
            9.90780317123359566267026821151376e-01 2.91419655191896319681998939188361e-03 6.30548632472147096983405845094239e-03 5.07778135112514102241995650466322e-05
            9.52058628185237454388811784156132e-01 1.51535125780546779672430801610972e-02 3.27878592367078641744981837291562e-02 5.98536868329913821634713499975078e-04
            8.84951337097152235244834628247190e-01 3.63650703847784376177010301489645e-02 7.86835925180693340763582455110736e-02 2.15009240011413945217744192461851e-03
            7.93658977143308685953115855227225e-01 6.52211475572483517781918749278702e-02 1.41119875299442976146480077659362e-01 4.89421074858147325120638626572145e-03
            6.83915749499090064311701553378953e-01 9.99092534147219268891859655923326e-02 2.16174997086188008799112481028715e-01 8.61626142643426642941939519459993e-03
            5.62616704255734512152287152275676e-01 1.38249971216943906560103982883447e-01 2.99133324527321553532033249211963e-01 1.27221626086710636049925682300454e-02
            4.37383295744265598870015310239978e-01 1.77834279283965807083589538706292e-01 3.84782424971768621801970766682643e-01 1.63648252403335824067820425398168e-02
            3.16084250500909769154844752847566e-01 2.16174997086187870021234402884147e-01 4.67740752412902305312769613010460e-01 1.86431208831233929945625504842610e-02
            2.06341022856691147513430450999294e-01 2.50863102943661486765591916991980e-01 5.42795874199647476743280094524380e-01 1.88248281551880655704245270953834e-02
            1.15048662902847709244014140494983e-01 2.79719180116131338476037626605830e-01 6.05232156981020841257645770383533e-01 1.65384550880889255597416820364742e-02
            4.79413718147626011223394471016945e-02 3.00930737922855129351518144176225e-01 6.51127890262382269526142408722080e-01 1.18862303728446382089511956792194e-02
            9.21968287664043373297317884862423e-03 3.13170053948990778636130016820971e-01 6.77610263174368787630896804330405e-01 5.45676666395715839752433851117530e-03
            9.90780317123359566267026821151376e-01 1.90239879518030769074699026788267e-03 7.31728408146012625906662307784245e-03 4.41829869967170135508158224091346e-05
            9.52058628185237454388811784156132e-01 9.89227169741105881450859982351176e-03 3.80491001173514850619561400435487e-02 5.20801209067746828304534822962069e-04
            8.84951337097152235244834628247190e-01 2.37392587816682536228096722652481e-02 9.13094041211795076629087475339475e-02 1.87084669439209765225140547073579e-03
            7.93658977143308685953115855227225e-01 4.25766177135455600732960590448783e-02 1.63764405143145747034694181820669e-01 4.25856953875835848172259900934478e-03
            6.83915749499090064311701553378953e-01 6.52211475572483517781918749278702e-02 2.50863102943661597787894379507634e-01 7.49721463449987482463576782265591e-03
            5.62616704255734512152287152275676e-01 9.02501166243023789093058439902961e-02 3.47133179119963108938407003734028e-01 1.10698572120377248390044044867864e-02
            4.37383295744265598870015310239978e-01 1.16090906232388740848548991380085e-01 4.46525798023345688037011314008851e-01 1.42394248747436009960853908751233e-02
            3.16084250500909769154844752847566e-01 1.41119875299442809613026383885881e-01 5.42795874199647365720977632008726e-01 1.62218242692696058837587003154113e-02
            2.06341022856691147513430450999294e-01 1.63764405143145608256816103676101e-01 6.29894572000163299740904676582431e-01 1.63799321018777074843164598405565e-02
            1.15048662902847709244014140494983e-01 1.82601764075022876543386018965975e-01 7.02349573022129414212599840539042e-01 1.43905043477484100716834447553083e-02
            4.79413718147626011223394471016945e-02 1.96448751159280093903092279106204e-01 7.55609877025957388241295120678842e-01 1.03424926299162094761596719649788e-02
            9.21968287664043373297317884862423e-03 2.04438624061510837437438681263302e-01 7.86341693061848756585163755516987e-01 4.74806286222457379908590269224078e-03
            9.90780317123359566267026821151376e-01 1.06071218734576249755063859225857e-03 8.15897068929467145226297475346655e-03 3.48123654591914581950436169410779e-05
            9.52058628185237454388811784156132e-01 5.51559072501670716315302911425533e-03 4.24257810897458384480351867296122e-02 4.10346227225478034708605612479460e-04
            8.84951337097152235244834628247190e-01 1.32361948357330866388181789261580e-02 1.01812468067114678116347192826652e-01 1.47406509315762906074664950750730e-03
            7.93658977143308685953115855227225e-01 2.37392587816682675005974800797048e-02 1.82601764075023043076839712739456e-01 3.35538380706700695155220159904275e-03
            6.83915749499090064311701553378953e-01 3.63650703847784584343827418706496e-02 2.79719180116131505009491320379311e-01 5.90715552575929881556904277317699e-03
            5.62616704255734512152287152275676e-01 5.03203633514185399988605240650941e-02 3.87062932392846947848852323659230e-01 8.72208832044691836915539795427321e-03
            4.37383295744265598870015310239978e-01 6.47282995514291553673658086154319e-02 4.97888404704305231884831073330133e-01 1.12194330072140462106755620652621e-02
            3.16084250500909769154844752847566e-01 7.86835925180692646874192064387898e-02 6.05232156981020952279948232899187e-01 1.27813919624437371397318585763969e-02
            2.06341022856691147513430450999294e-01 9.13094041211794660295453240905772e-02 7.02349573022129414212599840539042e-01 1.29059672350735135659682839559537e-02
            1.15048662902847709244014140494983e-01 1.01812468067114622605195961568825e-01 7.83138869030037598761850858863909e-01 1.13384705414580795601597884569856e-02
            4.79413718147626011223394471016945e-02 1.09533072177831009019755015287956e-01 8.42525556007406417613481153239263e-01 8.14898805321583041827437909887522e-03
            9.21968287664043373297317884862423e-03 1.13987950715501940024410032492597e-01 8.76792366407857626242616788658779e-01 3.74106213315225470672498175872533e-03
            9.90780317123359566267026821151376e-01 4.42004244803219056066806436078309e-04 8.77767863183721505637713278247247e-03 2.32561829723290052331172855781816e-05
            9.52058628185237454388811784156132e-01 2.29837513148131375451121982678160e-03 4.56429966832812283872300440634717e-02 2.74129230130811560144088456780764e-04
            8.84951337097152235244834628247190e-01 5.51559072501670716315302911425533e-03 1.09533072177831064530906246545783e-01 9.84739964303282724195076625051115e-04
            7.93658977143308685953115855227225e-01 9.89227169741106401867902775393304e-03 1.96448751159280260436545972879685e-01 2.24154316232875105638355783810312e-03
            6.83915749499090064311701553378953e-01 1.51535125780546849061369840683255e-02 3.00930737922855240373820606691879e-01 3.94623829610495253927826553308478e-03
            5.62616704255734512152287152275676e-01 2.09687552068421011186583058361066e-02 4.16414540537423383259607589934603e-01 5.82673654046600084099871708076535e-03
            4.37383295744265598870015310239978e-01 2.69726166079204895953402854047454e-02 5.35644087647813904595750500448048e-01 7.49507203603904086780529425482200e-03
            3.16084250500909769154844752847566e-01 3.27878592367079196856494149869832e-02 6.51127890262382269526142408722080e-01 8.53852894506958344511549086064406e-03
            2.06341022856691147513430450999294e-01 3.80491001173515475120012752086041e-02 7.55609877025957388241295120678842e-01 8.62175067665521890902802226719359e-03
            1.15048662902847709244014140494983e-01 4.24257810897458939591864179874392e-02 8.42525556007406306591178690723609e-01 7.57459431613802854771355654861509e-03
            4.79413718147626011223394471016945e-02 4.56429966832812908372751792285271e-02 9.06415631501956142734854893205920e-01 5.44388049203571935952261640068173e-03
            9.21968287664043373297317884862423e-03 4.74993675699593789762964490819286e-02 9.43280949553400138718473044718849e-01 2.49919315541567363392339018446364e-03
            9.90780317123359566267026821151376e-01 8.50025523458168270982820979853045e-05 9.13468032429461639087886482002432e-03 1.02592591132836700529643025880944e-05
            9.52058628185237454388811784156132e-01 4.42004244803219056066806436078309e-04 4.74993675699593234651452178241016e-02 1.20929681615560455604056755429809e-04
            8.84951337097152235244834628247190e-01 1.06071218734576249755063859225857e-03 1.13987950715501995535561263750424e-01 4.34409312354208166414432090718378e-04
            7.93658977143308685953115855227225e-01 1.90239879518030877494916275338710e-03 2.04438624061511003970892375036783e-01 9.88836910309057832632606555023358e-04
            6.83915749499090064311701553378953e-01 2.91419655191896493154346536869070e-03 3.13170053948990945169583710594452e-01 1.74084806826101936780282208871995e-03
            5.62616704255734512152287152275676e-01 4.03253528230196247456040836709690e-03 4.33350760461963524505790701368824e-01 2.57041321117075288391062848347701e-03
            4.37383295744265598870015310239978e-01 5.18714759433846952368929450472024e-03 5.57429556661395930738933657266898e-01 3.30638463680570787728285608864098e-03
            3.16084250500909769154844752847566e-01 6.30548632472147010247232046253885e-03 6.77610263174368676608594341814751e-01 3.76669640920738232198194950228753e-03
            2.06341022856691147513430450999294e-01 7.31728408146012625906662307784245e-03 7.86341693061848756585163755516987e-01 3.80340893891223689668446006351132e-03
            1.15048662902847709244014140494983e-01 8.15897068929467145226297475346655e-03 8.76792366407857515220314326143125e-01 3.34146518625723842541019870111541e-03
            4.79413718147626011223394471016945e-02 8.77767863183721505637713278247247e-03 9.43280949553400249740775507234503e-01 2.40151965677244151609048827822335e-03
            9.21968287664043373297317884862423e-03 9.13468032429461639087886482002432e-03 9.81645636799064913446954960818402e-01 1.10249692247698619258777785745451e-03
            ];
        
    case  23
        
        % ALG. DEG.:   23
        % PTS CARD.:  169
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.1e-15
        
        xyw_bar=[
            9.92091527359294067522910154366400e-01 7.84592870119713882426371043266045e-03 6.25439395087936528261352009394614e-05 6.48081397216732726002151945832175e-06
            9.58799199611488850436558095680084e-01 4.08749649858634253596356700199976e-02 3.25835402647724203806234299918287e-04 7.68281056147848793059368377456053e-05
            9.00789045366654939073214336531237e-01 9.84263475129689285880729698874347e-02 7.84607120376132338712693581328494e-04 2.78889724463864271085389301418900e-04
            8.21174669720170169462392095738323e-01 1.77411095047846606442831784988812e-01 1.41423523198322409477611927286489e-03 6.44849750211112164414994918359980e-04
            7.24246375518223350908897373301443e-01 2.73572834486986993240975607477594e-01 2.18078999478965585012701922096312e-03 1.15998876207291609120864084303548e-03
            6.15229157977567386872408405906754e-01 3.81727892345356867309646986541338e-01 3.04294967707574581794460755190812e-03 1.76241385864953103469743300735217e-03
            5.00000000000000000000000000000000e-01 4.96045763679647033761455077183200e-01 3.95423632035296623854492281679995e-03 2.35365454729507183814773974006584e-03
            3.84770842022432724149894056608900e-01 6.10363635013937089190960705309408e-01 4.86552296363018665914523808169179e-03 2.81801081538742784898810000981939e-03
            2.75753624481776538068800164182903e-01 7.18518692872307185304237009404460e-01 5.72768264591627662696282641263679e-03 3.04662416732331747457052095739982e-03
            1.78825330279829886048759135519504e-01 8.14680432311447377813351522490848e-01 6.49423740872268062673811073182151e-03 2.96118161683313037627063302181796e-03
            9.92109546333450609267856634687632e-02 8.93665179846325097301473761035595e-01 7.12386552032984177174057549564168e-03 2.53218820029313656377878416492422e-03
            4.12008003885109275188369792886078e-02 9.51216562373430885024561121099396e-01 7.58263723805818745660189961199649e-03 1.78789551359452938451843895251159e-03
            7.90847264070587696593861437577289e-03 9.84245598658096909616688208188862e-01 7.84592870119715790622194617753848e-03 8.12996506946889055736371609839352e-04
            9.92091527359294067522910154366400e-01 7.58263723805820567119839736847098e-03 3.25835402647726805891448265128929e-04 1.47471157249949821649477599283529e-05
            9.58799199611488850436558095680084e-01 3.95032944358572057907963426259812e-02 1.69750595265394377264556169393472e-03 1.74822633283280539284385013409917e-04
            9.00789045366654939073214336531237e-01 9.51233838951429794272840467783681e-02 4.08757073820208149950161669039517e-03 6.34614580644281820777374836239915e-04
            8.21174669720170169462392095738323e-01 1.71457583542560992873049485751835e-01 7.36774673726883766455841850984143e-03 1.46735794800435131492177820433653e-03
            7.24246375518223350908897373301443e-01 2.64392354443094501625921566301258e-01 1.13612700386821474651810603972990e-02 2.63955863992531430950450044292666e-03
            6.15229157977567386872408405906754e-01 3.68917975364947081651223470544210e-01 1.58528666574855314763681235490367e-02 4.01037913454377076966084203490936e-03
            5.00000000000000000000000000000000e-01 4.79399599805744425218279047840042e-01 2.06004001942555747817209521599580e-02 5.35574946830535796038486040515636e-03
            3.84770842022432724149894056608900e-01 5.89881224246541768785334625135874e-01 2.53479337310255070647713182552252e-02 6.41239384239077406246742896200885e-03
            2.75753624481776538068800164182903e-01 6.94406845168394459832938991894480e-01 2.98395303498290020982608439226169e-02 6.93260435479807653191830496552939e-03
            1.78825330279829886048759135519504e-01 7.87341616068927718785630531783681e-01 3.38330536512423396544591014389880e-02 6.73817952092244359246775431415699e-03
            9.92109546333450609267856634687632e-02 8.63675815716345884887061856716173e-01 3.71132296503090541861524798150640e-02 5.76200344394417300619704036535040e-03
            4.12008003885109275188369792886078e-02 9.19295905175631866690366678085411e-01 3.95032944358572057907963426259812e-02 4.06836273289300914823840571443725e-03
            7.90847264070587696593861437577289e-03 9.51216562373430662979956196068088e-01 4.08749649858634045429539582983125e-02 1.84997650348432294761802374694071e-03
            9.92091527359294067522910154366400e-01 7.12386552032980534254757998269270e-03 7.84607120376127134542265650907211e-04 2.22313328588331285266040809878163e-05
            9.58799199611488850436558095680084e-01 3.71132296503090611250463837222924e-02 4.08757073820208843839552059762354e-03 2.63545782392621134090221479695515e-04
            9.00789045366654939073214336531237e-01 8.93681411140854059516058782719483e-02 9.84281351925965497517978519681492e-03 9.56683886019795239920304741332302e-04
            8.21174669720170169462392095738323e-01 1.61083898550144682948470631345117e-01 1.77414317296851475891372729165596e-02 2.21204766908074086997904039719742e-03
            7.24246375518223350908897373301443e-01 2.48395844153334627302243120539060e-01 2.73577803284420217888595061594970e-02 3.97914465573290850736176693658308e-03
            6.15229157977567386872408405906754e-01 3.46597359470311117135565837088507e-01 3.81734825521214959920257570047397e-02 6.04566174787999309481190124415662e-03
            5.00000000000000000000000000000000e-01 4.50394522683327469536607168265618e-01 4.96054773166725304633928317343816e-02 8.07381262605881382110606381274920e-03
            3.84770842022432724149894056608900e-01 5.54191685896343821937648499442730e-01 6.10374720812234539124574439483695e-02 9.66670802552271717089649172294230e-03
            2.75753624481776538068800164182903e-01 6.52393201213320450548849294136744e-01 7.18531743049030113823505416803528e-02 1.04509273449920371545918129640995e-02
            1.78825330279829886048759135519504e-01 7.39705146816510117346865627041552e-01 8.14695229036599410932240061811171e-02 1.01578311708983726885380249882473e-02
            9.92109546333450609267856634687632e-02 8.11420904252569519243820650444832e-01 8.93681411140854198293936860864051e-02 8.68624203436291555946446862890298e-03
            4.12008003885109275188369792886078e-02 8.63675815716346106931666781747481e-01 9.51233838951429655494962389639113e-02 6.13307224219583960178603376789397e-03
            7.90847264070587696593861437577289e-03 8.93665179846325097301473761035595e-01 9.84263475129689702214363933308050e-02 2.78884659189817564473035105265808e-03
            9.92091527359294067522910154366400e-01 6.49423740872269623924939452308536e-03 1.41423523198323623784045111051455e-03 2.85182004077967694486393790409551e-05
            9.58799199611488850436558095680084e-01 3.38330536512423049599895819028461e-02 7.36774673726884460345232241706981e-03 3.38074711337701472660399737435455e-04
            9.00789045366654939073214336531237e-01 8.14695229036599133376483905522036e-02 1.77414317296851475891372729165596e-02 1.22722748841314131633406514509943e-03
            8.21174669720170169462392095738323e-01 1.46846831530139609878204964843462e-01 3.19784987496902206594029394182144e-02 2.83759948802976948514786847965752e-03
            7.24246375518223350908897373301443e-01 2.26441891507962761442840360359696e-01 4.93117329738138876482622663388611e-02 5.10441931054604267070651246740454e-03
            6.15229157977567386872408405906754e-01 3.15964069115722945291224732500268e-01 6.88067729067096678363668615929782e-02 7.75533317856327668315952195143836e-03
            5.00000000000000000000000000000000e-01 4.10587334860085084731196047869162e-01 8.94126651399149152688039521308383e-02 1.03570311320070138766391565354752e-02
            3.84770842022432724149894056608900e-01 5.05210600604447224171167363238055e-01 1.10018557373120051678938580153044e-01 1.24003863603697524892188752687616e-02
            2.75753624481776538068800164182903e-01 5.94732778212207491286278582265368e-01 1.29513597306015970644921253551729e-01 1.34063774927190692415335249165764e-02
            1.78825330279829886048759135519504e-01 6.74327838190030504073035899637034e-01 1.46846831530139554367053733585635e-01 1.30303957427881141745151793998048e-02
            9.92109546333450609267856634687632e-02 7.39705146816510228369168089557206e-01 1.61083898550144710704046246974031e-01 1.11426513515659707242999587606391e-02
            4.12008003885109275188369792886078e-02 7.87341616068928051852537919330643e-01 1.71457583542561020628625101380749e-01 7.86746275758909950626129869988290e-03
            7.90847264070587696593861437577289e-03 8.14680432311447488835653985006502e-01 1.77411095047846578687256169359898e-01 3.57751316663645554938533166478010e-03
            9.92091527359294067522910154366400e-01 5.72768264591630438253844204155030e-03 2.18078999478962809455140359204961e-03 3.32678832621699724640847162238089e-05
            9.58799199611488850436558095680084e-01 2.98395303498290090371547478298453e-02 1.13612700386821405262871564900706e-02 3.94380776831891783924116667847670e-04
            9.00789045366654939073214336531237e-01 7.18531743049030530157139651237230e-02 2.73577803284420079110716983450402e-02 1.43162121861982634396515123853533e-03
            8.21174669720170169462392095738323e-01 1.29513597306015942889345637922816e-01 4.93117329738138876482622663388611e-02 3.31019935208670375639816718660313e-03
            7.24246375518223350908897373301443e-01 1.99713563066939964185664280194032e-01 7.60400614148366849054383465045248e-02 5.95455615418094812846261376648727e-03
            6.15229157977567386872408405906754e-01 2.78668887739841775186278027831577e-01 1.06101954282590837941313566261670e-01 9.04697754957705865819228563395882e-03
            5.00000000000000000000000000000000e-01 3.62123187759111675454448686650721e-01 1.37876812240888324545551313349279e-01 1.20819861602511555154215727725386e-02
            3.84770842022432724149894056608900e-01 4.45577487778381520211468114212039e-01 1.69651670199185755638637829179061e-01 1.44656605235791921315602337472228e-02
            2.75753624481776538068800164182903e-01 5.24532812451283469989959939994151e-01 1.99713563066939991941239895822946e-01 1.56391986527461834821650654703262e-02
            1.78825330279829886048759135519504e-01 5.94732778212207269241673657234060e-01 2.26441891507962789198415975988610e-01 1.52005974511784415492687827509144e-02
            9.92109546333450609267856634687632e-02 6.52393201213320339526546831621090e-01 2.48395844153334599546667504910147e-01 1.29984507821051444714788303258501e-02
            4.12008003885109275188369792886078e-02 6.94406845168394459832938991894480e-01 2.64392354443094612648224028816912e-01 9.17778221789152318099702654308203e-03
            7.90847264070587696593861437577289e-03 7.18518692872307074281934546888806e-01 2.73572834486986993240975607477594e-01 4.17334504613408284467990227994960e-03
            9.92091527359294067522910154366400e-01 4.86552296363013982161138670790024e-03 3.04294967707579265547845892569967e-03 3.62241632174357529750513551292102e-05
            9.58799199611488850436558095680084e-01 2.53479337310255452286877897449813e-02 1.58528666574856043347541145749346e-02 4.29426588917449208321341913574543e-04
            9.00789045366654939073214336531237e-01 6.10374720812235163625025791134249e-02 3.81734825521215445642830843553384e-02 1.55883920477140167255847291727378e-03
            8.21174669720170169462392095738323e-01 1.10018557373120093312302003596415e-01 6.88067729067097372253059006652620e-02 3.60435320357732624785285224788822e-03
            7.24246375518223350908897373301443e-01 1.69651670199185755638637829179061e-01 1.06101954282590893452464797519497e-01 6.48369516980106669334915636682126e-03
            6.15229157977567386872408405906754e-01 2.36722241151780865564546729729045e-01 1.48048600870651747563044864364201e-01 9.85091804001298788906471060045078e-03
            5.00000000000000000000000000000000e-01 3.07614578988783693436204202953377e-01 1.92385421011216306563795797046623e-01 1.31556262600396774165023572322752e-02
            3.84770842022432724149894056608900e-01 3.78506916825786521307861676177708e-01 2.36722241151780754542244267213391e-01 1.57511207949324223698273073068776e-02
            2.75753624481776538068800164182903e-01 4.45577487778381686744921807985520e-01 2.78668887739841775186278027831577e-01 1.70289429033552187908728825505023e-02
            1.78825330279829886048759135519504e-01 5.05210600604447224171167363238055e-01 3.15964069115722834268922269984614e-01 1.65513663353557723134645840445955e-02
            9.92109546333450609267856634687632e-02 5.54191685896343821937648499442730e-01 3.46597359470311117135565837088507e-01 1.41535305686313565148859439091211e-02
            4.12008003885109275188369792886078e-02 5.89881224246541990829939550167182e-01 3.68917975364947081651223470544210e-01 9.99334638801713769085122152091571e-03
            7.90847264070587696593861437577289e-03 6.10363635013937200213263167825062e-01 3.81727892345356867309646986541338e-01 4.54420051081955069655071710599259e-03
            9.92091527359294067522910154366400e-01 3.95423632035296623854492281679995e-03 3.95423632035296623854492281679995e-03 3.72276251859123723849526177431812e-05
            9.58799199611488850436558095680084e-01 2.06004001942555747817209521599580e-02 2.06004001942555747817209521599580e-02 4.41322329549047570173420096395489e-04
            9.00789045366654939073214336531237e-01 4.96054773166725304633928317343816e-02 4.96054773166725304633928317343816e-02 1.60202131632409920658499657974971e-03
            8.21174669720170169462392095738323e-01 8.94126651399149152688039521308383e-02 8.94126651399149152688039521308383e-02 3.70419902579926813707134058972770e-03
            7.24246375518223350908897373301443e-01 1.37876812240888324545551313349279e-01 1.37876812240888324545551313349279e-01 6.66330294925583915127687006929591e-03
            6.15229157977567386872408405906754e-01 1.92385421011216334319371412675537e-01 1.92385421011216278808220181417710e-01 1.01238027868157873245813505036494e-02
            5.00000000000000000000000000000000e-01 2.50000000000000000000000000000000e-01 2.50000000000000000000000000000000e-01 1.35200562275230393854963395483537e-02
            3.84770842022432724149894056608900e-01 3.07614578988783637925052971695550e-01 3.07614578988783637925052971695550e-01 1.61874497332635119972721327030740e-02
            2.75753624481776538068800164182903e-01 3.62123187759111730965599917908548e-01 3.62123187759111730965599917908548e-01 1.75006693712464571344167296729211e-02
            1.78825330279829886048759135519504e-01 4.10587334860085029220044816611335e-01 4.10587334860085029220044816611335e-01 1.70098632382147847696973741449256e-02
            9.92109546333450609267856634687632e-02 4.50394522683327469536607168265618e-01 4.50394522683327469536607168265618e-01 1.45456039357935475175853667906267e-02
            4.12008003885109275188369792886078e-02 4.79399599805744536240581510355696e-01 4.79399599805744536240581510355696e-01 1.02701766070616897635892073026298e-02
            7.90847264070587696593861437577289e-03 4.96045763679647033761455077183200e-01 4.96045763679647033761455077183200e-01 4.67008146940425025034349104657849e-03
            9.92091527359294067522910154366400e-01 3.04294967707579352284019691410322e-03 4.86552296363013895424964871949669e-03 3.62241632174356310023069505099613e-05
            9.58799199611488850436558095680084e-01 1.58528666574856078042010665285488e-02 2.53479337310255417592408377913671e-02 4.29426588917447798858517682418778e-04
            9.00789045366654939073214336531237e-01 3.81734825521215515031769882625667e-02 6.10374720812235094236086752061965e-02 1.55883920477139646838804498685249e-03
            8.21174669720170169462392095738323e-01 6.88067729067097649808815162941755e-02 1.10018557373120065556726387967501e-01 3.60435320357731410478852041023856e-03
            7.24246375518223350908897373301443e-01 1.06101954282590921208040413148410e-01 1.69651670199185727883062213550147e-01 6.48369516980104500930570665673258e-03
            6.15229157977567386872408405906754e-01 1.48048600870651858585347326879855e-01 2.36722241151780754542244267213391e-01 9.85091804001295492931866704111599e-03
            5.00000000000000000000000000000000e-01 1.92385421011216362074947028304450e-01 3.07614578988783637925052971695550e-01 1.31556262600396340484154578120979e-02
            3.84770842022432724149894056608900e-01 2.36722241151780837808971114100132e-01 3.78506916825786410285559213662054e-01 1.57511207949323703281230280026648e-02
            2.75753624481776538068800164182903e-01 2.78668887739841830697429259089404e-01 4.45577487778381631233770576727693e-01 1.70289429033551598102746993390610e-02
            1.78825330279829886048759135519504e-01 3.15964069115722945291224732500268e-01 5.05210600604447113148864900722401e-01 1.65513663353557168023133527867685e-02
            9.92109546333450609267856634687632e-02 3.46597359470311172646717068346334e-01 5.54191685896343821937648499442730e-01 1.41535305686313096773520925353296e-02
            4.12008003885109275188369792886078e-02 3.68917975364947192673525933059864e-01 5.89881224246541879807637087651528e-01 9.99334638801710473110517796158092e-03
            7.90847264070587696593861437577289e-03 3.81727892345356922820798217799165e-01 6.10363635013937200213263167825062e-01 4.54420051081953508403943331472874e-03
            9.92091527359294067522910154366400e-01 2.18078999478962722718966560364606e-03 5.72768264591630524990018002995384e-03 3.32678832621699385827668260517953e-05
            9.58799199611488850436558095680084e-01 1.13612700386821370568402045364564e-02 2.98395303498290125066016997834595e-02 3.94380776831891404453356297921118e-04
            9.00789045366654939073214336531237e-01 2.73577803284420009721777944378118e-02 7.18531743049030668935017729381798e-02 1.43162121861982504292254425593001e-03
            8.21174669720170169462392095738323e-01 4.93117329738138598926866507099476e-02 1.29513597306015970644921253551729e-01 3.31019935208670072063208422719072e-03
            7.24246375518223350908897373301443e-01 7.60400614148366571498627308756113e-02 1.99713563066939991941239895822946e-01 5.95455615418094205693044784766244e-03
            6.15229157977567386872408405906754e-01 1.06101954282590865696889181890583e-01 2.78668887739841775186278027831577e-01 9.04697754957704998457490574992335e-03
            5.00000000000000000000000000000000e-01 1.37876812240888269034400082091452e-01 3.62123187759111730965599917908548e-01 1.20819861602511451070807169116961e-02
            3.84770842022432724149894056608900e-01 1.69651670199185672371910982292320e-01 4.45577487778381575722619345469866e-01 1.44656605235791782537724259327661e-02
            2.75753624481776538068800164182903e-01 1.99713563066939908674513048936205e-01 5.24532812451283581012262402509805e-01 1.56391986527461696043772576558695e-02
            1.78825330279829886048759135519504e-01 2.26441891507962650420537897844042e-01 5.94732778212207380263976119749714e-01 1.52005974511784276714809749364576e-02
            9.92109546333450609267856634687632e-02 2.48395844153334544035516273652320e-01 6.52393201213320450548849294136744e-01 1.29984507821051323284144984882005e-02
            4.12008003885109275188369792886078e-02 2.64392354443094446114770335043431e-01 6.94406845168394681877543916925788e-01 9.17778221789151624210312263585365e-03
            7.90847264070587696593861437577289e-03 2.73572834486986937729824376219767e-01 7.18518692872307074281934546888806e-01 4.17334504613407850787121233793187e-03
            9.92091527359294067522910154366400e-01 1.41423523198323688836175460181721e-03 6.49423740872269537188765653468181e-03 2.85182004077968202706162142989754e-05
            9.58799199611488850436558095680084e-01 7.36774673726884980762275034749109e-03 3.38330536512422980210956779956177e-02 3.38074711337702068971594604462894e-04
            9.00789045366654939073214336531237e-01 1.77414317296851510585842248701738e-02 8.14695229036599133376483905522036e-02 1.22722748841314348473841011610830e-03
            8.21174669720170169462392095738323e-01 3.19784987496902345371907472326711e-02 1.46846831530139609878204964843462e-01 2.83759948802977425563742741587703e-03
            7.24246375518223350908897373301443e-01 4.93117329738138945871561702460895e-02 2.26441891507962761442840360359696e-01 5.10441931054605134432389235144001e-03
            6.15229157977567386872408405906754e-01 6.88067729067097649808815162941755e-02 3.15964069115722834268922269984614e-01 7.75533317856328969358559177749157e-03
            5.00000000000000000000000000000000e-01 8.94126651399149430243795677597518e-02 4.10587334860085029220044816611335e-01 1.03570311320070312238739163035461e-02
            3.84770842022432724149894056608900e-01 1.10018557373120107190089811410871e-01 5.05210600604447224171167363238055e-01 1.24003863603697750406240629672538e-02
            2.75753624481776538068800164182903e-01 1.29513597306016026156072484809556e-01 5.94732778212207380263976119749714e-01 1.34063774927190917929387126150687e-02
            1.78825330279829886048759135519504e-01 1.46846831530139637633780580472376e-01 6.74327838190030393050733437121380e-01 1.30303957427881367259203670982970e-02
            9.92109546333450609267856634687632e-02 1.61083898550144738459621862602944e-01 7.39705146816510228369168089557206e-01 1.11426513515659898062581945055172e-02
            4.12008003885109275188369792886078e-02 1.71457583542561076139776332638576e-01 7.87341616068927940830235456814989e-01 7.86746275758911338404910651433966e-03
            7.90847264070587696593861437577289e-03 1.77411095047846661953983016246639e-01 8.14680432311447377813351522490848e-01 3.57751316663646162091749758360493e-03
            9.92091527359294067522910154366400e-01 7.84607120376126917701831153806324e-04 7.12386552032980534254757998269270e-03 2.22313328588331556316583931254272e-05
            9.58799199611488850436558095680084e-01 4.08757073820208496894856864400936e-03 3.71132296503090680639402876295208e-02 2.63545782392621459350873225346845e-04
            9.00789045366654939073214336531237e-01 9.84281351925965150573283324320073e-03 8.93681411140854059516058782719483e-02 9.56683886019796324122477226836736e-04
            8.21174669720170169462392095738323e-01 1.77414317296851441196903209629454e-02 1.61083898550144682948470631345117e-01 2.21204766908074347206425436240806e-03
            7.24246375518223350908897373301443e-01 2.73577803284420113805186502986544e-02 2.48395844153334627302243120539060e-01 3.97914465573291371153219486700436e-03
            6.15229157977567386872408405906754e-01 3.81734825521215445642830843553384e-02 3.46597359470311061624414605830680e-01 6.04566174788000003370580515138499e-03
            5.00000000000000000000000000000000e-01 4.96054773166725304633928317343816e-02 4.50394522683327469536607168265618e-01 8.07381262605882422944691967359176e-03
            3.84770842022432724149894056608900e-01 6.10374720812235094236086752061965e-02 5.54191685896343821937648499442730e-01 9.66670802552272757923734758378487e-03
            2.75753624481776538068800164182903e-01 7.18531743049030530157139651237230e-02 6.52393201213320450548849294136744e-01 1.04509273449920492976561448017492e-02
            1.78825330279829886048759135519504e-01 8.14695229036598994598605827377469e-02 7.39705146816510117346865627041552e-01 1.01578311708983848316023568258970e-02
            9.92109546333450609267856634687632e-02 8.93681411140854059516058782719483e-02 8.11420904252569519243820650444832e-01 8.68624203436292596780532448974554e-03
            4.12008003885109275188369792886078e-02 9.51233838951429933050718545928248e-02 8.63675815716346106931666781747481e-01 6.13307224219584654067993767512235e-03
            7.90847264070587696593861437577289e-03 9.84263475129689285880729698874347e-02 8.93665179846325097301473761035595e-01 2.78884659189817911417730300627227e-03
            9.92091527359294067522910154366400e-01 3.25835402647725016957863664046613e-04 7.58263723805820740592187334527807e-03 1.47471157249950092700020720659637e-05
            9.58799199611488850436558095680084e-01 1.69750595265393141274079535918418e-03 3.95032944358572196685841504404380e-02 1.74822633283280864545036759061247e-04
            9.00789045366654939073214336531237e-01 4.08757073820206328490511893392068e-03 9.51233838951429933050718545928248e-02 6.34614580644282904979547321744349e-04
            8.21174669720170169462392095738323e-01 7.36774673726880817425932690412083e-03 1.71457583542561020628625101380749e-01 1.46735794800435391700699216954717e-03
            7.24246375518223350908897373301443e-01 1.13612700386820798109654973018223e-02 2.64392354443094557137072797559085e-01 2.63955863992531907999405937914617e-03
            6.15229157977567386872408405906754e-01 1.58528666574855210680272676881941e-02 3.68917975364947081651223470544210e-01 4.01037913454377770855474594213774e-03
            5.00000000000000000000000000000000e-01 2.06004001942554637594184896443039e-02 4.79399599805744536240581510355696e-01 5.35574946830536836872571626599893e-03
            3.84770842022432724149894056608900e-01 2.53479337310254029813627596467995e-02 5.89881224246541879807637087651528e-01 6.41239384239078620553176079965851e-03
            2.75753624481776538068800164182903e-01 2.98395303498288529120419099172068e-02 6.94406845168394570855241454410134e-01 6.93260435479808954234437479158260e-03
            1.78825330279829886048759135519504e-01 3.38330536512421176098541764076799e-02 7.87341616068927940830235456814989e-01 6.73817952092245573553208615180665e-03
            9.92109546333450609267856634687632e-02 3.71132296503088668360170743198978e-02 8.63675815716346106931666781747481e-01 5.76200344394418341453789622619297e-03
            4.12008003885109275188369792886078e-02 3.95032944358570045628731293163582e-02 9.19295905175632088734971603116719e-01 4.06836273289301608713230962166563e-03
            7.90847264070587696593861437577289e-03 4.08749649858632033150307449886895e-02 9.51216562373430885024561121099396e-01 1.84997650348432641706497570055490e-03
            9.92091527359294067522910154366400e-01 6.25439395087938290089882298339319e-05 7.84592870119713882426371043266045e-03 6.48081397216735267100993708733192e-06
            9.58799199611488850436558095680084e-01 3.25835402647724474856777421294396e-04 4.08749649858634253596356700199976e-02 7.68281056147851910140614273281301e-05
            9.00789045366654939073214336531237e-01 7.84607120376121388270751477733711e-04 9.84263475129689424658607777018915e-02 2.78889724463865409497670411198555e-04
            8.21174669720170169462392095738323e-01 1.41423523198322648002089874097464e-03 1.77411095047846606442831784988812e-01 6.44849750211114658079991635020178e-04
            7.24246375518223350908897373301443e-01 2.18078999478961291572098879498753e-03 2.73572834486987048752126838735421e-01 1.15998876207292064485776528215411e-03
            6.15229157977567386872408405906754e-01 3.04294967707577183879674720401454e-03 3.81727892345356867309646986541338e-01 1.76241385864953797359133691458055e-03
            5.00000000000000000000000000000000e-01 3.95423632035293848296930718788644e-03 4.96045763679647033761455077183200e-01 2.35365454729508137912685761250486e-03
            3.84770842022432724149894056608900e-01 4.86552296363010425978012918335480e-03 6.10363635013937200213263167825062e-01 2.81801081538743869100982486486373e-03
            2.75753624481776538068800164182903e-01 5.72768264591626448389849457498713e-03 7.18518692872307185304237009404460e-01 3.04662416732332961763485279504948e-03
            1.78825330279829886048759135519504e-01 6.49423740872264940171554314929381e-03 8.14680432311447377813351522490848e-01 2.96118161683314208565409586526584e-03
            9.92109546333450609267856634687632e-02 7.12386552032975590292851464369051e-03 8.93665179846325208323776223551249e-01 2.53218820029314653843877103156501e-03
            4.12008003885109275188369792886078e-02 7.58263723805815449685585605266169e-03 9.51216562373430885024561121099396e-01 1.78789551359453654025277735684085e-03
            7.90847264070587696593861437577289e-03 7.84592870119708331311247917483342e-03 9.84245598658097020638990670704516e-01 8.12996506946892199922671817802211e-04
            ];
        
    case  24
        
        % ALG. DEG.:   24
        % PTS CARD.:  169
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.1e-15
        
        xyw_bar=[
            9.92091527359294067522910154366400e-01 7.84592870119713882426371043266045e-03 6.25439395087936528261352009394614e-05 6.48081397216732726002151945832175e-06
            9.58799199611488850436558095680084e-01 4.08749649858634253596356700199976e-02 3.25835402647724203806234299918287e-04 7.68281056147848793059368377456053e-05
            9.00789045366654939073214336531237e-01 9.84263475129689285880729698874347e-02 7.84607120376132338712693581328494e-04 2.78889724463864271085389301418900e-04
            8.21174669720170169462392095738323e-01 1.77411095047846606442831784988812e-01 1.41423523198322409477611927286489e-03 6.44849750211112164414994918359980e-04
            7.24246375518223350908897373301443e-01 2.73572834486986993240975607477594e-01 2.18078999478965585012701922096312e-03 1.15998876207291609120864084303548e-03
            6.15229157977567386872408405906754e-01 3.81727892345356867309646986541338e-01 3.04294967707574581794460755190812e-03 1.76241385864953103469743300735217e-03
            5.00000000000000000000000000000000e-01 4.96045763679647033761455077183200e-01 3.95423632035296623854492281679995e-03 2.35365454729507183814773974006584e-03
            3.84770842022432724149894056608900e-01 6.10363635013937089190960705309408e-01 4.86552296363018665914523808169179e-03 2.81801081538742784898810000981939e-03
            2.75753624481776538068800164182903e-01 7.18518692872307185304237009404460e-01 5.72768264591627662696282641263679e-03 3.04662416732331747457052095739982e-03
            1.78825330279829886048759135519504e-01 8.14680432311447377813351522490848e-01 6.49423740872268062673811073182151e-03 2.96118161683313037627063302181796e-03
            9.92109546333450609267856634687632e-02 8.93665179846325097301473761035595e-01 7.12386552032984177174057549564168e-03 2.53218820029313656377878416492422e-03
            4.12008003885109275188369792886078e-02 9.51216562373430885024561121099396e-01 7.58263723805818745660189961199649e-03 1.78789551359452938451843895251159e-03
            7.90847264070587696593861437577289e-03 9.84245598658096909616688208188862e-01 7.84592870119715790622194617753848e-03 8.12996506946889055736371609839352e-04
            9.92091527359294067522910154366400e-01 7.58263723805820567119839736847098e-03 3.25835402647726805891448265128929e-04 1.47471157249949821649477599283529e-05
            9.58799199611488850436558095680084e-01 3.95032944358572057907963426259812e-02 1.69750595265394377264556169393472e-03 1.74822633283280539284385013409917e-04
            9.00789045366654939073214336531237e-01 9.51233838951429794272840467783681e-02 4.08757073820208149950161669039517e-03 6.34614580644281820777374836239915e-04
            8.21174669720170169462392095738323e-01 1.71457583542560992873049485751835e-01 7.36774673726883766455841850984143e-03 1.46735794800435131492177820433653e-03
            7.24246375518223350908897373301443e-01 2.64392354443094501625921566301258e-01 1.13612700386821474651810603972990e-02 2.63955863992531430950450044292666e-03
            6.15229157977567386872408405906754e-01 3.68917975364947081651223470544210e-01 1.58528666574855314763681235490367e-02 4.01037913454377076966084203490936e-03
            5.00000000000000000000000000000000e-01 4.79399599805744425218279047840042e-01 2.06004001942555747817209521599580e-02 5.35574946830535796038486040515636e-03
            3.84770842022432724149894056608900e-01 5.89881224246541768785334625135874e-01 2.53479337310255070647713182552252e-02 6.41239384239077406246742896200885e-03
            2.75753624481776538068800164182903e-01 6.94406845168394459832938991894480e-01 2.98395303498290020982608439226169e-02 6.93260435479807653191830496552939e-03
            1.78825330279829886048759135519504e-01 7.87341616068927718785630531783681e-01 3.38330536512423396544591014389880e-02 6.73817952092244359246775431415699e-03
            9.92109546333450609267856634687632e-02 8.63675815716345884887061856716173e-01 3.71132296503090541861524798150640e-02 5.76200344394417300619704036535040e-03
            4.12008003885109275188369792886078e-02 9.19295905175631866690366678085411e-01 3.95032944358572057907963426259812e-02 4.06836273289300914823840571443725e-03
            7.90847264070587696593861437577289e-03 9.51216562373430662979956196068088e-01 4.08749649858634045429539582983125e-02 1.84997650348432294761802374694071e-03
            9.92091527359294067522910154366400e-01 7.12386552032980534254757998269270e-03 7.84607120376127134542265650907211e-04 2.22313328588331285266040809878163e-05
            9.58799199611488850436558095680084e-01 3.71132296503090611250463837222924e-02 4.08757073820208843839552059762354e-03 2.63545782392621134090221479695515e-04
            9.00789045366654939073214336531237e-01 8.93681411140854059516058782719483e-02 9.84281351925965497517978519681492e-03 9.56683886019795239920304741332302e-04
            8.21174669720170169462392095738323e-01 1.61083898550144682948470631345117e-01 1.77414317296851475891372729165596e-02 2.21204766908074086997904039719742e-03
            7.24246375518223350908897373301443e-01 2.48395844153334627302243120539060e-01 2.73577803284420217888595061594970e-02 3.97914465573290850736176693658308e-03
            6.15229157977567386872408405906754e-01 3.46597359470311117135565837088507e-01 3.81734825521214959920257570047397e-02 6.04566174787999309481190124415662e-03
            5.00000000000000000000000000000000e-01 4.50394522683327469536607168265618e-01 4.96054773166725304633928317343816e-02 8.07381262605881382110606381274920e-03
            3.84770842022432724149894056608900e-01 5.54191685896343821937648499442730e-01 6.10374720812234539124574439483695e-02 9.66670802552271717089649172294230e-03
            2.75753624481776538068800164182903e-01 6.52393201213320450548849294136744e-01 7.18531743049030113823505416803528e-02 1.04509273449920371545918129640995e-02
            1.78825330279829886048759135519504e-01 7.39705146816510117346865627041552e-01 8.14695229036599410932240061811171e-02 1.01578311708983726885380249882473e-02
            9.92109546333450609267856634687632e-02 8.11420904252569519243820650444832e-01 8.93681411140854198293936860864051e-02 8.68624203436291555946446862890298e-03
            4.12008003885109275188369792886078e-02 8.63675815716346106931666781747481e-01 9.51233838951429655494962389639113e-02 6.13307224219583960178603376789397e-03
            7.90847264070587696593861437577289e-03 8.93665179846325097301473761035595e-01 9.84263475129689702214363933308050e-02 2.78884659189817564473035105265808e-03
            9.92091527359294067522910154366400e-01 6.49423740872269623924939452308536e-03 1.41423523198323623784045111051455e-03 2.85182004077967694486393790409551e-05
            9.58799199611488850436558095680084e-01 3.38330536512423049599895819028461e-02 7.36774673726884460345232241706981e-03 3.38074711337701472660399737435455e-04
            9.00789045366654939073214336531237e-01 8.14695229036599133376483905522036e-02 1.77414317296851475891372729165596e-02 1.22722748841314131633406514509943e-03
            8.21174669720170169462392095738323e-01 1.46846831530139609878204964843462e-01 3.19784987496902206594029394182144e-02 2.83759948802976948514786847965752e-03
            7.24246375518223350908897373301443e-01 2.26441891507962761442840360359696e-01 4.93117329738138876482622663388611e-02 5.10441931054604267070651246740454e-03
            6.15229157977567386872408405906754e-01 3.15964069115722945291224732500268e-01 6.88067729067096678363668615929782e-02 7.75533317856327668315952195143836e-03
            5.00000000000000000000000000000000e-01 4.10587334860085084731196047869162e-01 8.94126651399149152688039521308383e-02 1.03570311320070138766391565354752e-02
            3.84770842022432724149894056608900e-01 5.05210600604447224171167363238055e-01 1.10018557373120051678938580153044e-01 1.24003863603697524892188752687616e-02
            2.75753624481776538068800164182903e-01 5.94732778212207491286278582265368e-01 1.29513597306015970644921253551729e-01 1.34063774927190692415335249165764e-02
            1.78825330279829886048759135519504e-01 6.74327838190030504073035899637034e-01 1.46846831530139554367053733585635e-01 1.30303957427881141745151793998048e-02
            9.92109546333450609267856634687632e-02 7.39705146816510228369168089557206e-01 1.61083898550144710704046246974031e-01 1.11426513515659707242999587606391e-02
            4.12008003885109275188369792886078e-02 7.87341616068928051852537919330643e-01 1.71457583542561020628625101380749e-01 7.86746275758909950626129869988290e-03
            7.90847264070587696593861437577289e-03 8.14680432311447488835653985006502e-01 1.77411095047846578687256169359898e-01 3.57751316663645554938533166478010e-03
            9.92091527359294067522910154366400e-01 5.72768264591630438253844204155030e-03 2.18078999478962809455140359204961e-03 3.32678832621699724640847162238089e-05
            9.58799199611488850436558095680084e-01 2.98395303498290090371547478298453e-02 1.13612700386821405262871564900706e-02 3.94380776831891783924116667847670e-04
            9.00789045366654939073214336531237e-01 7.18531743049030530157139651237230e-02 2.73577803284420079110716983450402e-02 1.43162121861982634396515123853533e-03
            8.21174669720170169462392095738323e-01 1.29513597306015942889345637922816e-01 4.93117329738138876482622663388611e-02 3.31019935208670375639816718660313e-03
            7.24246375518223350908897373301443e-01 1.99713563066939964185664280194032e-01 7.60400614148366849054383465045248e-02 5.95455615418094812846261376648727e-03
            6.15229157977567386872408405906754e-01 2.78668887739841775186278027831577e-01 1.06101954282590837941313566261670e-01 9.04697754957705865819228563395882e-03
            5.00000000000000000000000000000000e-01 3.62123187759111675454448686650721e-01 1.37876812240888324545551313349279e-01 1.20819861602511555154215727725386e-02
            3.84770842022432724149894056608900e-01 4.45577487778381520211468114212039e-01 1.69651670199185755638637829179061e-01 1.44656605235791921315602337472228e-02
            2.75753624481776538068800164182903e-01 5.24532812451283469989959939994151e-01 1.99713563066939991941239895822946e-01 1.56391986527461834821650654703262e-02
            1.78825330279829886048759135519504e-01 5.94732778212207269241673657234060e-01 2.26441891507962789198415975988610e-01 1.52005974511784415492687827509144e-02
            9.92109546333450609267856634687632e-02 6.52393201213320339526546831621090e-01 2.48395844153334599546667504910147e-01 1.29984507821051444714788303258501e-02
            4.12008003885109275188369792886078e-02 6.94406845168394459832938991894480e-01 2.64392354443094612648224028816912e-01 9.17778221789152318099702654308203e-03
            7.90847264070587696593861437577289e-03 7.18518692872307074281934546888806e-01 2.73572834486986993240975607477594e-01 4.17334504613408284467990227994960e-03
            9.92091527359294067522910154366400e-01 4.86552296363013982161138670790024e-03 3.04294967707579265547845892569967e-03 3.62241632174357529750513551292102e-05
            9.58799199611488850436558095680084e-01 2.53479337310255452286877897449813e-02 1.58528666574856043347541145749346e-02 4.29426588917449208321341913574543e-04
            9.00789045366654939073214336531237e-01 6.10374720812235163625025791134249e-02 3.81734825521215445642830843553384e-02 1.55883920477140167255847291727378e-03
            8.21174669720170169462392095738323e-01 1.10018557373120093312302003596415e-01 6.88067729067097372253059006652620e-02 3.60435320357732624785285224788822e-03
            7.24246375518223350908897373301443e-01 1.69651670199185755638637829179061e-01 1.06101954282590893452464797519497e-01 6.48369516980106669334915636682126e-03
            6.15229157977567386872408405906754e-01 2.36722241151780865564546729729045e-01 1.48048600870651747563044864364201e-01 9.85091804001298788906471060045078e-03
            5.00000000000000000000000000000000e-01 3.07614578988783693436204202953377e-01 1.92385421011216306563795797046623e-01 1.31556262600396774165023572322752e-02
            3.84770842022432724149894056608900e-01 3.78506916825786521307861676177708e-01 2.36722241151780754542244267213391e-01 1.57511207949324223698273073068776e-02
            2.75753624481776538068800164182903e-01 4.45577487778381686744921807985520e-01 2.78668887739841775186278027831577e-01 1.70289429033552187908728825505023e-02
            1.78825330279829886048759135519504e-01 5.05210600604447224171167363238055e-01 3.15964069115722834268922269984614e-01 1.65513663353557723134645840445955e-02
            9.92109546333450609267856634687632e-02 5.54191685896343821937648499442730e-01 3.46597359470311117135565837088507e-01 1.41535305686313565148859439091211e-02
            4.12008003885109275188369792886078e-02 5.89881224246541990829939550167182e-01 3.68917975364947081651223470544210e-01 9.99334638801713769085122152091571e-03
            7.90847264070587696593861437577289e-03 6.10363635013937200213263167825062e-01 3.81727892345356867309646986541338e-01 4.54420051081955069655071710599259e-03
            9.92091527359294067522910154366400e-01 3.95423632035296623854492281679995e-03 3.95423632035296623854492281679995e-03 3.72276251859123723849526177431812e-05
            9.58799199611488850436558095680084e-01 2.06004001942555747817209521599580e-02 2.06004001942555747817209521599580e-02 4.41322329549047570173420096395489e-04
            9.00789045366654939073214336531237e-01 4.96054773166725304633928317343816e-02 4.96054773166725304633928317343816e-02 1.60202131632409920658499657974971e-03
            8.21174669720170169462392095738323e-01 8.94126651399149152688039521308383e-02 8.94126651399149152688039521308383e-02 3.70419902579926813707134058972770e-03
            7.24246375518223350908897373301443e-01 1.37876812240888324545551313349279e-01 1.37876812240888324545551313349279e-01 6.66330294925583915127687006929591e-03
            6.15229157977567386872408405906754e-01 1.92385421011216334319371412675537e-01 1.92385421011216278808220181417710e-01 1.01238027868157873245813505036494e-02
            5.00000000000000000000000000000000e-01 2.50000000000000000000000000000000e-01 2.50000000000000000000000000000000e-01 1.35200562275230393854963395483537e-02
            3.84770842022432724149894056608900e-01 3.07614578988783637925052971695550e-01 3.07614578988783637925052971695550e-01 1.61874497332635119972721327030740e-02
            2.75753624481776538068800164182903e-01 3.62123187759111730965599917908548e-01 3.62123187759111730965599917908548e-01 1.75006693712464571344167296729211e-02
            1.78825330279829886048759135519504e-01 4.10587334860085029220044816611335e-01 4.10587334860085029220044816611335e-01 1.70098632382147847696973741449256e-02
            9.92109546333450609267856634687632e-02 4.50394522683327469536607168265618e-01 4.50394522683327469536607168265618e-01 1.45456039357935475175853667906267e-02
            4.12008003885109275188369792886078e-02 4.79399599805744536240581510355696e-01 4.79399599805744536240581510355696e-01 1.02701766070616897635892073026298e-02
            7.90847264070587696593861437577289e-03 4.96045763679647033761455077183200e-01 4.96045763679647033761455077183200e-01 4.67008146940425025034349104657849e-03
            9.92091527359294067522910154366400e-01 3.04294967707579352284019691410322e-03 4.86552296363013895424964871949669e-03 3.62241632174356310023069505099613e-05
            9.58799199611488850436558095680084e-01 1.58528666574856078042010665285488e-02 2.53479337310255417592408377913671e-02 4.29426588917447798858517682418778e-04
            9.00789045366654939073214336531237e-01 3.81734825521215515031769882625667e-02 6.10374720812235094236086752061965e-02 1.55883920477139646838804498685249e-03
            8.21174669720170169462392095738323e-01 6.88067729067097649808815162941755e-02 1.10018557373120065556726387967501e-01 3.60435320357731410478852041023856e-03
            7.24246375518223350908897373301443e-01 1.06101954282590921208040413148410e-01 1.69651670199185727883062213550147e-01 6.48369516980104500930570665673258e-03
            6.15229157977567386872408405906754e-01 1.48048600870651858585347326879855e-01 2.36722241151780754542244267213391e-01 9.85091804001295492931866704111599e-03
            5.00000000000000000000000000000000e-01 1.92385421011216362074947028304450e-01 3.07614578988783637925052971695550e-01 1.31556262600396340484154578120979e-02
            3.84770842022432724149894056608900e-01 2.36722241151780837808971114100132e-01 3.78506916825786410285559213662054e-01 1.57511207949323703281230280026648e-02
            2.75753624481776538068800164182903e-01 2.78668887739841830697429259089404e-01 4.45577487778381631233770576727693e-01 1.70289429033551598102746993390610e-02
            1.78825330279829886048759135519504e-01 3.15964069115722945291224732500268e-01 5.05210600604447113148864900722401e-01 1.65513663353557168023133527867685e-02
            9.92109546333450609267856634687632e-02 3.46597359470311172646717068346334e-01 5.54191685896343821937648499442730e-01 1.41535305686313096773520925353296e-02
            4.12008003885109275188369792886078e-02 3.68917975364947192673525933059864e-01 5.89881224246541879807637087651528e-01 9.99334638801710473110517796158092e-03
            7.90847264070587696593861437577289e-03 3.81727892345356922820798217799165e-01 6.10363635013937200213263167825062e-01 4.54420051081953508403943331472874e-03
            9.92091527359294067522910154366400e-01 2.18078999478962722718966560364606e-03 5.72768264591630524990018002995384e-03 3.32678832621699385827668260517953e-05
            9.58799199611488850436558095680084e-01 1.13612700386821370568402045364564e-02 2.98395303498290125066016997834595e-02 3.94380776831891404453356297921118e-04
            9.00789045366654939073214336531237e-01 2.73577803284420009721777944378118e-02 7.18531743049030668935017729381798e-02 1.43162121861982504292254425593001e-03
            8.21174669720170169462392095738323e-01 4.93117329738138598926866507099476e-02 1.29513597306015970644921253551729e-01 3.31019935208670072063208422719072e-03
            7.24246375518223350908897373301443e-01 7.60400614148366571498627308756113e-02 1.99713563066939991941239895822946e-01 5.95455615418094205693044784766244e-03
            6.15229157977567386872408405906754e-01 1.06101954282590865696889181890583e-01 2.78668887739841775186278027831577e-01 9.04697754957704998457490574992335e-03
            5.00000000000000000000000000000000e-01 1.37876812240888269034400082091452e-01 3.62123187759111730965599917908548e-01 1.20819861602511451070807169116961e-02
            3.84770842022432724149894056608900e-01 1.69651670199185672371910982292320e-01 4.45577487778381575722619345469866e-01 1.44656605235791782537724259327661e-02
            2.75753624481776538068800164182903e-01 1.99713563066939908674513048936205e-01 5.24532812451283581012262402509805e-01 1.56391986527461696043772576558695e-02
            1.78825330279829886048759135519504e-01 2.26441891507962650420537897844042e-01 5.94732778212207380263976119749714e-01 1.52005974511784276714809749364576e-02
            9.92109546333450609267856634687632e-02 2.48395844153334544035516273652320e-01 6.52393201213320450548849294136744e-01 1.29984507821051323284144984882005e-02
            4.12008003885109275188369792886078e-02 2.64392354443094446114770335043431e-01 6.94406845168394681877543916925788e-01 9.17778221789151624210312263585365e-03
            7.90847264070587696593861437577289e-03 2.73572834486986937729824376219767e-01 7.18518692872307074281934546888806e-01 4.17334504613407850787121233793187e-03
            9.92091527359294067522910154366400e-01 1.41423523198323688836175460181721e-03 6.49423740872269537188765653468181e-03 2.85182004077968202706162142989754e-05
            9.58799199611488850436558095680084e-01 7.36774673726884980762275034749109e-03 3.38330536512422980210956779956177e-02 3.38074711337702068971594604462894e-04
            9.00789045366654939073214336531237e-01 1.77414317296851510585842248701738e-02 8.14695229036599133376483905522036e-02 1.22722748841314348473841011610830e-03
            8.21174669720170169462392095738323e-01 3.19784987496902345371907472326711e-02 1.46846831530139609878204964843462e-01 2.83759948802977425563742741587703e-03
            7.24246375518223350908897373301443e-01 4.93117329738138945871561702460895e-02 2.26441891507962761442840360359696e-01 5.10441931054605134432389235144001e-03
            6.15229157977567386872408405906754e-01 6.88067729067097649808815162941755e-02 3.15964069115722834268922269984614e-01 7.75533317856328969358559177749157e-03
            5.00000000000000000000000000000000e-01 8.94126651399149430243795677597518e-02 4.10587334860085029220044816611335e-01 1.03570311320070312238739163035461e-02
            3.84770842022432724149894056608900e-01 1.10018557373120107190089811410871e-01 5.05210600604447224171167363238055e-01 1.24003863603697750406240629672538e-02
            2.75753624481776538068800164182903e-01 1.29513597306016026156072484809556e-01 5.94732778212207380263976119749714e-01 1.34063774927190917929387126150687e-02
            1.78825330279829886048759135519504e-01 1.46846831530139637633780580472376e-01 6.74327838190030393050733437121380e-01 1.30303957427881367259203670982970e-02
            9.92109546333450609267856634687632e-02 1.61083898550144738459621862602944e-01 7.39705146816510228369168089557206e-01 1.11426513515659898062581945055172e-02
            4.12008003885109275188369792886078e-02 1.71457583542561076139776332638576e-01 7.87341616068927940830235456814989e-01 7.86746275758911338404910651433966e-03
            7.90847264070587696593861437577289e-03 1.77411095047846661953983016246639e-01 8.14680432311447377813351522490848e-01 3.57751316663646162091749758360493e-03
            9.92091527359294067522910154366400e-01 7.84607120376126917701831153806324e-04 7.12386552032980534254757998269270e-03 2.22313328588331556316583931254272e-05
            9.58799199611488850436558095680084e-01 4.08757073820208496894856864400936e-03 3.71132296503090680639402876295208e-02 2.63545782392621459350873225346845e-04
            9.00789045366654939073214336531237e-01 9.84281351925965150573283324320073e-03 8.93681411140854059516058782719483e-02 9.56683886019796324122477226836736e-04
            8.21174669720170169462392095738323e-01 1.77414317296851441196903209629454e-02 1.61083898550144682948470631345117e-01 2.21204766908074347206425436240806e-03
            7.24246375518223350908897373301443e-01 2.73577803284420113805186502986544e-02 2.48395844153334627302243120539060e-01 3.97914465573291371153219486700436e-03
            6.15229157977567386872408405906754e-01 3.81734825521215445642830843553384e-02 3.46597359470311061624414605830680e-01 6.04566174788000003370580515138499e-03
            5.00000000000000000000000000000000e-01 4.96054773166725304633928317343816e-02 4.50394522683327469536607168265618e-01 8.07381262605882422944691967359176e-03
            3.84770842022432724149894056608900e-01 6.10374720812235094236086752061965e-02 5.54191685896343821937648499442730e-01 9.66670802552272757923734758378487e-03
            2.75753624481776538068800164182903e-01 7.18531743049030530157139651237230e-02 6.52393201213320450548849294136744e-01 1.04509273449920492976561448017492e-02
            1.78825330279829886048759135519504e-01 8.14695229036598994598605827377469e-02 7.39705146816510117346865627041552e-01 1.01578311708983848316023568258970e-02
            9.92109546333450609267856634687632e-02 8.93681411140854059516058782719483e-02 8.11420904252569519243820650444832e-01 8.68624203436292596780532448974554e-03
            4.12008003885109275188369792886078e-02 9.51233838951429933050718545928248e-02 8.63675815716346106931666781747481e-01 6.13307224219584654067993767512235e-03
            7.90847264070587696593861437577289e-03 9.84263475129689285880729698874347e-02 8.93665179846325097301473761035595e-01 2.78884659189817911417730300627227e-03
            9.92091527359294067522910154366400e-01 3.25835402647725016957863664046613e-04 7.58263723805820740592187334527807e-03 1.47471157249950092700020720659637e-05
            9.58799199611488850436558095680084e-01 1.69750595265393141274079535918418e-03 3.95032944358572196685841504404380e-02 1.74822633283280864545036759061247e-04
            9.00789045366654939073214336531237e-01 4.08757073820206328490511893392068e-03 9.51233838951429933050718545928248e-02 6.34614580644282904979547321744349e-04
            8.21174669720170169462392095738323e-01 7.36774673726880817425932690412083e-03 1.71457583542561020628625101380749e-01 1.46735794800435391700699216954717e-03
            7.24246375518223350908897373301443e-01 1.13612700386820798109654973018223e-02 2.64392354443094557137072797559085e-01 2.63955863992531907999405937914617e-03
            6.15229157977567386872408405906754e-01 1.58528666574855210680272676881941e-02 3.68917975364947081651223470544210e-01 4.01037913454377770855474594213774e-03
            5.00000000000000000000000000000000e-01 2.06004001942554637594184896443039e-02 4.79399599805744536240581510355696e-01 5.35574946830536836872571626599893e-03
            3.84770842022432724149894056608900e-01 2.53479337310254029813627596467995e-02 5.89881224246541879807637087651528e-01 6.41239384239078620553176079965851e-03
            2.75753624481776538068800164182903e-01 2.98395303498288529120419099172068e-02 6.94406845168394570855241454410134e-01 6.93260435479808954234437479158260e-03
            1.78825330279829886048759135519504e-01 3.38330536512421176098541764076799e-02 7.87341616068927940830235456814989e-01 6.73817952092245573553208615180665e-03
            9.92109546333450609267856634687632e-02 3.71132296503088668360170743198978e-02 8.63675815716346106931666781747481e-01 5.76200344394418341453789622619297e-03
            4.12008003885109275188369792886078e-02 3.95032944358570045628731293163582e-02 9.19295905175632088734971603116719e-01 4.06836273289301608713230962166563e-03
            7.90847264070587696593861437577289e-03 4.08749649858632033150307449886895e-02 9.51216562373430885024561121099396e-01 1.84997650348432641706497570055490e-03
            9.92091527359294067522910154366400e-01 6.25439395087938290089882298339319e-05 7.84592870119713882426371043266045e-03 6.48081397216735267100993708733192e-06
            9.58799199611488850436558095680084e-01 3.25835402647724474856777421294396e-04 4.08749649858634253596356700199976e-02 7.68281056147851910140614273281301e-05
            9.00789045366654939073214336531237e-01 7.84607120376121388270751477733711e-04 9.84263475129689424658607777018915e-02 2.78889724463865409497670411198555e-04
            8.21174669720170169462392095738323e-01 1.41423523198322648002089874097464e-03 1.77411095047846606442831784988812e-01 6.44849750211114658079991635020178e-04
            7.24246375518223350908897373301443e-01 2.18078999478961291572098879498753e-03 2.73572834486987048752126838735421e-01 1.15998876207292064485776528215411e-03
            6.15229157977567386872408405906754e-01 3.04294967707577183879674720401454e-03 3.81727892345356867309646986541338e-01 1.76241385864953797359133691458055e-03
            5.00000000000000000000000000000000e-01 3.95423632035293848296930718788644e-03 4.96045763679647033761455077183200e-01 2.35365454729508137912685761250486e-03
            3.84770842022432724149894056608900e-01 4.86552296363010425978012918335480e-03 6.10363635013937200213263167825062e-01 2.81801081538743869100982486486373e-03
            2.75753624481776538068800164182903e-01 5.72768264591626448389849457498713e-03 7.18518692872307185304237009404460e-01 3.04662416732332961763485279504948e-03
            1.78825330279829886048759135519504e-01 6.49423740872264940171554314929381e-03 8.14680432311447377813351522490848e-01 2.96118161683314208565409586526584e-03
            9.92109546333450609267856634687632e-02 7.12386552032975590292851464369051e-03 8.93665179846325208323776223551249e-01 2.53218820029314653843877103156501e-03
            4.12008003885109275188369792886078e-02 7.58263723805815449685585605266169e-03 9.51216562373430885024561121099396e-01 1.78789551359453654025277735684085e-03
            7.90847264070587696593861437577289e-03 7.84592870119708331311247917483342e-03 9.84245598658097020638990670704516e-01 8.12996506946892199922671817802211e-04
            ];
        
    case  25
        
        % ALG. DEG.:   25
        % PTS CARD.:  196
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.4e-15
        
        xyw_bar=[
            9.93141904348405990532455689390190e-01 6.81106217562754169547734051093357e-03 4.70334759664677720669700988764816e-05 4.22930698511761701580309591452789e-06
            9.64217441831786703509976632631151e-01 3.55371579616369642762307989869441e-02 2.45400206576332213792568381904857e-04 5.03658965179998215902154590217776e-05
            9.13600657534882620858240898087388e-01 8.58068075102567801026864913183090e-02 5.92534954860599039072610594303114e-04 1.84361795037071271368897540376963e-04
            8.43646452405842683930359271471389e-01 1.55281260009290517087876537516422e-01 1.07228758486679898176419101218926e-03 4.31605399414997873558386132231135e-04
            7.57624318179077160806400570436381e-01 2.40713446211374693106677113974001e-01 1.66223560954814608692231558961794e-03 7.89660988168044432854164593749147e-04
            6.59556184463944927820477914792718e-01 3.38109019285115164343835658655735e-01 2.33479625093990783568642655154690e-03 1.22669725145557045875577273363888e-03
            5.54027474353671722795411369588692e-01 4.42914003407462730788068938636570e-01 3.05852223886554641651969177473802e-03 1.68576507847577523359350326614958e-03
            4.45972525646328055159983705380000e-01 5.50227900940943426277840444527101e-01 3.79957341272851856217585009289905e-03 2.09421010280399578010190886345754e-03
            3.40443815536054961157219622691628e-01 6.55032885063290937210922493250109e-01 4.52329940065410163185788405826315e-03 2.37653240194270719276570247302516e-03
            2.42375681820922894704750660821446e-01 7.52428458137031408448081037931843e-01 5.19586004204575235831953250453807e-03 2.46834320695364043546371490833735e-03
            1.56353547594157371580791959786438e-01 8.37860644339115445689003536244854e-01 5.78580806672723824135573522653431e-03 2.32883979710401135126973670708139e-03
            8.63993424651176011863640269439202e-02 9.07335096838148946751800849597203e-01 6.26556069673345206183512345887721e-03 1.94947151638545776095035044761516e-03
            3.57825581682132409788721361110220e-02 9.57604746386769178911890776362270e-01 6.61269544501763562038831878453493e-03 1.35718848462018214923174852515331e-03
            6.85809565159406497869554186763708e-03 9.86330842172778443632807920948835e-01 6.81106217562754689964776844135486e-03 6.12458940011641954186127900783276e-04
            9.93141904348405990532455689390190e-01 6.61269544501762260996224895848172e-03 2.45400206576386857582061651328331e-04 9.65314258066566015356515378931235e-06
            9.64217441831786703509976632631151e-01 3.45021666991517303446634912234003e-02 1.28039146906156614535987614544865e-03 1.14957174308259263894801960326220e-04
            9.13600657534882620858240898087388e-01 8.33077529676639338962473857463920e-02 3.09158949745344524551171616622014e-03 4.20794872583791677248671314615081e-04
            8.43646452405842683930359271471389e-01 1.50758817682562884510133471849258e-01 5.59472991159443155950725667935330e-03 9.85113748847970963420928036669011e-04
            7.57624318179077160806400570436381e-01 2.33702859887605363109486233952339e-01 8.67282193331747608411319561128039e-03 1.80235441314588896305481924287051e-03
            6.59556184463944927820477914792718e-01 3.28261864903627686995690737603582e-01 1.21819506324273851838313476037001e-02 2.79986378697028677128844087462767e-03
            5.54027474353671722795411369588692e-01 4.30014487805963463440406258087023e-01 1.59580378403648137641823723242851e-02 3.84765889950671033034601542510700e-03
            4.45972525646328055159983705380000e-01 5.34202954025823406603024068317609e-01 1.98245203278485382369922263023909e-02 4.77991046461605229023206220517750e-03
            3.40443815536054961157219622691628e-01 6.35955576928159072025437126285397e-01 2.36006075357859668173432510229759e-02 5.42429438303987203340295053521913e-03
            2.42375681820922894704750660821446e-01 7.30514581944181395911641629936639e-01 2.71097362348957648947589404997416e-02 5.63384710511345872252508826250050e-03
            1.56353547594157371580791959786438e-01 8.13458624149223874510994392039720e-01 3.01878282566188094193648794316687e-02 5.31543883858038787582067641324102e-03
            8.63993424651176011863640269439202e-02 8.80909688864122575324699937482364e-01 3.26909686707598234889360355737153e-02 4.44955321778136951571536172878041e-03
            3.57825581682132409788721361110220e-02 9.29715275132635077248721700016176e-01 3.45021666991517372835573951306287e-02 3.09770229424758585315413661476214e-03
            6.85809565159406497869554186763708e-03 9.57604746386769067889588313846616e-01 3.55371579616369226428673755435739e-02 1.39790123855748410139354653836108e-03
            9.93141904348405990532455689390190e-01 6.26556069673335578468220674608347e-03 5.92534954860653682862103863726588e-04 1.46340329542932308327842644279926e-05
            9.64217441831786703509976632631151e-01 3.26909686707598512445116512026289e-02 3.09158949745344524551171616622014e-03 1.74273513842938620284003792271221e-04
            9.13600657534882620858240898087388e-01 7.89344960867127409809285154551617e-02 7.46484637840463816083058645745041e-03 6.37919307720842868568433825515740e-04
            8.43646452405842683930359271471389e-01 1.42844703889933699603176364689716e-01 1.35088437042236164664643638388952e-02 1.49341905435467457718967665414311e-03
            7.57624318179077160806400570436381e-01 2.21434582282060643532162202973268e-01 2.09410995388621956614372265903512e-02 2.73234479413169122768989538485584e-03
            6.59556184463944927820477914792718e-01 3.11029693727424227933653355648858e-01 2.94141218086308442458687295584241e-02 4.24455544747894854273884135409389e-03
            5.54027474353671722795411369588692e-01 4.07440792672977780686238702401170e-01 3.85317329733504965183499280101387e-02 5.83299859726903874301839891813870e-03
            4.45972525646328055159983705380000e-01 5.06159864861905006705455889459699e-01 4.78676094917669381345604051603004e-02 7.24627929953758267295205186542262e-03
            3.40443815536054961157219622691628e-01 6.02570963807458559458041236212011e-01 5.69852206564864793847391410963610e-02 8.22315656190373243938918079720679e-03
            2.42375681820922894704750660821446e-01 6.92166075252822032837229926371947e-01 6.54582429262551279691706440644339e-02 8.54083564048989558881164185777379e-03
            1.56353547594157371580791959786438e-01 7.70755953644948976766215764655499e-01 7.28904987608937071641435068158899e-02 8.05813303598293569074417774800168e-03
            8.63993424651176011863640269439202e-02 8.34666161448169630077131841972005e-01 7.89344960867127687365041310840752e-02 6.74546220329461532899673059660017e-03
            3.57825581682132409788721361110220e-02 8.80909688864122908391607325029327e-01 8.33077529676639061406717701174784e-02 4.69607457652230445704910621884665e-03
            6.85809565159406497869554186763708e-03 9.07335096838149168796405774628511e-01 8.58068075102568217360499147616792e-02 2.11919927846822286898742504490656e-03
            9.93141904348405990532455689390190e-01 5.78580806672717492394886207307536e-03 1.07228758486683454359544853673469e-03 1.89313971987104344007080830580847e-05
            9.64217441831786703509976632631151e-01 3.01878282566188545221752548286531e-02 5.59472991159444196784811254019587e-03 2.25449889451542057321881129539065e-04
            9.13600657534882620858240898087388e-01 7.28904987608937487975069302592601e-02 1.35088437042236303442521716533520e-02 8.25247820126486994474213076955493e-04
            8.43646452405842683930359271471389e-01 1.31907115748878894789797300290957e-01 2.44464318452784212798434282376547e-02 1.93196977145090268562777069405456e-03
            7.57624318179077160806400570436381e-01 2.04479384117668888665164672602259e-01 3.78962977032539505284347569613601e-02 3.53471286712938540658379693581992e-03
            6.59556184463944927820477914792718e-01 2.87214217220501943916843856641208e-01 5.32295983155531282626782285660738e-02 5.49099249394542111846195808766424e-03
            5.54027474353671722795411369588692e-01 3.76243139131998471924589466652833e-01 6.97293865143298052799991637584753e-02 7.54589070896035697677550757589415e-03
            4.45972525646328055159983705380000e-01 4.67403313273844323028072267334210e-01 8.66241610798276218119440272857901e-02 9.37418906058248499468810166490584e-03
            3.40443815536054961157219622691628e-01 5.56432235185340795524666646088008e-01 1.03123949278604243318113731220365e-01 1.06379317025461911372374856910028e-02
            2.42375681820922894704750660821446e-01 6.39167068288173823020770214498043e-01 1.18457249890903337785630355938338e-01 1.10488989893644563855534812546466e-02
            1.56353547594157371580791959786438e-01 7.11739336656963761384986355551518e-01 1.31907115748878922545372915919870e-01 1.04244481108325508234013767605575e-02
            8.63993424651176011863640269439202e-02 7.70755953644948754721610839624191e-01 1.42844703889933644092025133431889e-01 8.72630427021108684426664581224031e-03
            3.57825581682132409788721361110220e-02 8.13458624149223874510994392039720e-01 1.50758817682562940021284703107085e-01 6.07510269797689825971298560602918e-03
            6.85809565159406497869554186763708e-03 8.37860644339115445689003536244854e-01 1.55281260009290544843452153145336e-01 2.74151379932027110600256669670216e-03
            9.93141904348405990532455689390190e-01 5.19586004204576363402212635378419e-03 1.66223560954824583352218425602587e-03 2.23437044034388537338663843767605e-05
            9.64217441831786703509976632631151e-01 2.71097362348957683642058924533558e-02 8.67282193331752812581747491549322e-03 2.66086313377670777551714520470227e-04
            9.13600657534882620858240898087388e-01 6.54582429262551418469584518788906e-02 2.09410995388622372948006500337215e-02 9.73995377041929756394000872177230e-04
            8.43646452405842683930359271471389e-01 1.18457249890903323907842548123881e-01 3.78962977032539921617981804047304e-02 2.28019945049902361677851558852126e-03
            7.57624318179077160806400570436381e-01 1.83629710682765651119652261513693e-01 5.87459711381571880739471680499264e-02 4.17183045842758372379055487044752e-03
            6.59556184463944927820477914792718e-01 2.57928513623787214470439721480943e-01 8.25153019122678577090823637263384e-02 6.48072151666520025747875166644008e-03
            5.54027474353671722795411369588692e-01 3.37879630669400410347691376955481e-01 1.08092894976927866856897253455827e-01 8.90600676178041061881174300651764e-03
            4.45972525646328055159983705380000e-01 4.19744687509676861481011655996554e-01 1.34282786843995083359004638623446e-01 1.10638484414596078436732184968605e-02
            3.40443815536054961157219622691628e-01 4.99695804555290001847112080213265e-01 1.59860379908655036995668297095108e-01 1.25553755452267462588977764426090e-02
            2.42375681820922894704750660821446e-01 5.73994607496311592953475155809429e-01 1.83629710682765567852925414626952e-01 1.30404180109131191678262240429831e-02
            1.56353547594157371580791959786438e-01 6.39167068288173823020770214498043e-01 2.04479384117668860909589056973346e-01 1.23034124059947964913153128918566e-02
            8.63993424651176011863640269439202e-02 6.92166075252821810792625001340639e-01 2.21434582282060588021010971715441e-01 1.02991850575795962363656599336537e-02
            3.57825581682132409788721361110220e-02 7.30514581944181506933944092452293e-01 2.33702859887605307598335002694512e-01 7.17011520488176207210884172127408e-03
            6.85809565159406497869554186763708e-03 7.52428458137031408448081037931843e-01 2.40713446211374582084374651458347e-01 3.23566049071820870108862067127120e-03
            9.93141904348405990532455689390190e-01 4.52329940065408081517617233657802e-03 2.33479625093992865236813827323203e-03 2.47112936176651894551459243132285e-05
            9.64217441831786703509976632631151e-01 2.36006075357859286534267795332198e-02 1.21819506324273678365965878356292e-02 2.94281418102975984418118171603851e-04
            9.13600657534882620858240898087388e-01 5.69852206564865071403147567252745e-02 2.94141218086308720014443451873376e-02 1.07720211965511391931371587560307e-03
            8.43646452405842683930359271471389e-01 1.03123949278604215562538115591451e-01 5.32295983155531005071026129371603e-02 2.52181451699871406538222551318995e-03
            7.57624318179077160806400570436381e-01 1.59860379908655036995668297095108e-01 8.25153019122678021979311324685114e-02 4.61388700458534046361380731582358e-03
            6.59556184463944927820477914792718e-01 2.24541823999307582626627777244721e-01 1.15901991536747489552894307962561e-01 7.16743335666338628880955852196166e-03
            5.54027474353671722795411369588692e-01 2.94143937391041054851825720106717e-01 1.51828588255287222352762910304591e-01 9.84970728566367768885836397885214e-03
            4.45972525646328055159983705380000e-01 3.65412247072903983990954657201655e-01 1.88615227280767960849061637418345e-01 1.22361987270193754706593125547442e-02
            3.40443815536054961157219622691628e-01 4.35014360464637428460576984434738e-01 2.24541823999307610382203392873635e-01 1.38857714001264769682064681433076e-02
            2.42375681820922894704750660821446e-01 4.99695804555289946335960848955438e-01 2.57928513623787214470439721480943e-01 1.44222100573066867174043537147554e-02
            1.56353547594157371580791959786438e-01 5.56432235185340684502364183572354e-01 2.87214217220501999427995087899035e-01 1.36071096794929437268084626566633e-02
            8.63993424651176011863640269439202e-02 6.02570963807458226391133848665049e-01 3.11029693727424172422502124391031e-01 1.13905098897275538355478019525435e-02
            3.57825581682132409788721361110220e-02 6.35955576928159072025437126285397e-01 3.28261864903627742506841968861409e-01 7.92987675190731135244082139479360e-03
            6.85809565159406497869554186763708e-03 6.55032885063290826188620030734455e-01 3.38109019285115164343835658655735e-01 3.57851835978058698331527587299661e-03
            9.93141904348405990532455689390190e-01 3.79957341272849687813240038281037e-03 3.05852223886551258941191022699968e-03 2.59234313538678390104099541479954e-05
            9.64217441831786703509976632631151e-01 1.98245203278485590536739380240761e-02 1.59580378403647374363494293447729e-02 3.08716502622017469059656491481292e-04
            9.13600657534882620858240898087388e-01 4.78676094917669173178786934386153e-02 3.85317329733504618238804084739968e-02 1.13004101020263434188017903636592e-03
            8.43646452405842683930359271471389e-01 8.66241610798275801785806038424198e-02 6.97293865143297358910601246861916e-02 2.64551449754415286472197799128026e-03
            7.57624318179077160806400570436381e-01 1.34282786843995055603429022994533e-01 1.08092894976927783590170406569086e-01 4.84020727075039954717672330275491e-03
            6.59556184463944927820477914792718e-01 1.88615227280767905337910406160518e-01 1.51828588255287166841611679046764e-01 7.51901011252851213251258144509848e-03
            5.54027474353671722795411369588692e-01 2.47081032014963314358269030890369e-01 1.98891493631364962846319599520939e-01 1.03328548730068289773775802586897e-02
            4.45972525646328055159983705380000e-01 3.06946442338708491703869185585063e-01 2.47081032014963453136147109034937e-01 1.28364084308971292819334664159214e-02
            3.40443815536054961157219622691628e-01 3.65412247072903872968652194686001e-01 2.94143937391041165874128182622371e-01 1.45668959001544577602782482017574e-02
            2.42375681820922894704750660821446e-01 4.19744687509676694947557962223073e-01 3.37879630669400465858842608213308e-01 1.51296479324896284562873560730623e-02
            1.56353547594157371580791959786438e-01 4.67403313273844156494618573560729e-01 3.76243139131998527435740697910660e-01 1.42745652719986730605494784640541e-02
            8.63993424651176011863640269439202e-02 5.06159864861904673638548501912737e-01 4.07440792672977725175087471143343e-01 1.19492368866039224556407916111311e-02
            3.57825581682132409788721361110220e-02 5.34202954025823184558419143286301e-01 4.30014487805963629973859951860504e-01 8.31885286150084793388437276462355e-03
            6.85809565159406497869554186763708e-03 5.50227900940943204233235519495793e-01 4.42914003407462786299220169894397e-01 3.75405174992586186319409868872299e-03
            9.93141904348405990532455689390190e-01 3.05852223886545534353720299236556e-03 3.79957341272855412400710761744449e-03 2.59234313538678762798596333372103e-05
            9.64217441831786703509976632631151e-01 1.59580378403647304974555254375446e-02 1.98245203278485659925678419313044e-02 3.08716502622017902740525485683065e-04
            9.13600657534882620858240898087388e-01 3.85317329733504410071986967523117e-02 4.78676094917669381345604051603004e-02 1.13004101020263585976322051607212e-03
            8.43646452405842683930359271471389e-01 6.97293865143296942576967012428213e-02 8.66241610798276218119440272857901e-02 2.64551449754415633416892994489444e-03
            7.57624318179077160806400570436381e-01 1.08092894976927783590170406569086e-01 1.34282786843995055603429022994533e-01 4.84020727075040648607062720998329e-03
            6.59556184463944927820477914792718e-01 1.51828588255287111330460447788937e-01 1.88615227280767960849061637418345e-01 7.51901011252852340821517529434459e-03
            5.54027474353671722795411369588692e-01 1.98891493631364796312865905747458e-01 2.47081032014963480891722724663850e-01 1.03328548730068445898888640499536e-02
            4.45972525646328055159983705380000e-01 2.47081032014963314358269030890369e-01 3.06946442338708602726171648100717e-01 1.28364084308971483638917021607995e-02
            3.40443815536054961157219622691628e-01 2.94143937391040999340674488848890e-01 3.65412247072904039502105888459482e-01 1.45668959001544785769599599234425e-02
            2.42375681820922894704750660821446e-01 3.37879630669400299325388914439827e-01 4.19744687509676861481011655996554e-01 1.51296479324896492729690677947474e-02
            1.56353547594157371580791959786438e-01 3.76243139131998360902287004137179e-01 4.67403313273844323028072267334210e-01 1.42745652719986921425077142089322e-02
            8.63993424651176011863640269439202e-02 4.07440792672977503130482546112034e-01 5.06159864861904895683153426944045e-01 1.19492368866039398028755513792021e-02
            3.57825581682132409788721361110220e-02 4.30014487805963352418103795571369e-01 5.34202954025823517625326530833263e-01 8.31885286150086007694870460227321e-03
            6.85809565159406497869554186763708e-03 4.42914003407462564254615244863089e-01 5.50227900940943426277840444527101e-01 3.75405174992586706736452661914427e-03
            9.93141904348405990532455689390190e-01 2.33479625093987270753603802120324e-03 4.52329940065413632632740359440504e-03 2.47112936176650776467968867455838e-05
            9.64217441831786703509976632631151e-01 1.21819506324273643671496358820150e-02 2.36006075357859321228737314868340e-02 2.94281418102974683375511188998530e-04
            9.13600657534882620858240898087388e-01 2.94141218086308615931034893264950e-02 5.69852206564865210181025645397312e-02 1.07720211965510914882415693938356e-03
            8.43646452405842683930359271471389e-01 5.32295983155530866293148051227035e-02 1.03123949278604229440325923405908e-01 2.52181451699870278967963166394384e-03
            7.57624318179077160806400570436381e-01 8.25153019122678299535067480974249e-02 1.59860379908655009240092681466194e-01 4.61388700458532051429383358254199e-03
            6.59556184463944927820477914792718e-01 1.15901991536747461797318692333647e-01 2.24541823999307610382203392873635e-01 7.16743335666335419642525295103042e-03
            5.54027474353671722795411369588692e-01 1.51828588255287111330460447788937e-01 2.94143937391041165874128182622371e-01 9.84970728566363432077146455867478e-03
            4.45972525646328055159983705380000e-01 1.88615227280767905337910406160518e-01 3.65412247072904039502105888459482e-01 1.22361987270193216942315572737243e-02
            3.40443815536054961157219622691628e-01 2.24541823999307554871052161615808e-01 4.35014360464637483971728215692565e-01 1.38857714001264145181613329782522e-02
            2.42375681820922894704750660821446e-01 2.57928513623787158959288490223116e-01 4.99695804555290001847112080213265e-01 1.44222100573066225326357425728929e-02
            1.56353547594157371580791959786438e-01 2.87214217220501888405692625383381e-01 5.56432235185340795524666646088008e-01 1.36071096794928830114868034684150e-02
            8.63993424651176011863640269439202e-02 3.11029693727424005889048430617549e-01 6.02570963807458337413436311180703e-01 1.13905098897275017938435226483307e-02
            3.57825581682132409788721361110220e-02 3.28261864903627631484539506345754e-01 6.35955576928159183047739588801051e-01 7.92987675190727665797130185865171e-03
            6.85809565159406497869554186763708e-03 3.38109019285115053321533196140081e-01 6.55032885063290937210922493250109e-01 3.57851835978057093712312308753098e-03
            9.93141904348405990532455689390190e-01 1.66223560954819097289225648950151e-03 5.19586004204581827781161962320766e-03 2.23437044034388943914478525831768e-05
            9.64217441831786703509976632631151e-01 8.67282193331753159526442686910741e-03 2.71097362348957648947589404997416e-02 2.66086313377671265442692138947223e-04
            9.13600657534882620858240898087388e-01 2.09410995388622407642476019873357e-02 6.54582429262551418469584518788906e-02 9.73995377041931599537694097534768e-04
            8.43646452405842683930359271471389e-01 3.78962977032539991006920843119588e-02 1.18457249890903310030054740309424e-01 2.28019945049902795358720553053899e-03
            7.57624318179077160806400570436381e-01 5.87459711381572505239923032149818e-02 1.83629710682765595608501030255866e-01 4.17183045842759153004619676607945e-03
            6.59556184463944927820477914792718e-01 8.25153019122678438312945559118816e-02 2.57928513623787214470439721480943e-01 6.48072151666521153318134551568619e-03
            5.54027474353671722795411369588692e-01 1.08092894976927825223533830012457e-01 3.37879630669400465858842608213308e-01 8.90600676178042796604650277458859e-03
            4.45972525646328055159983705380000e-01 1.34282786843995111114580254252360e-01 4.19744687509676861481011655996554e-01 1.10638484414596269256314542417385e-02
            3.40443815536054961157219622691628e-01 1.59860379908655064751243912724021e-01 4.99695804555289946335960848955438e-01 1.25553755452267688103029641411013e-02
            2.42375681820922894704750660821446e-01 1.83629710682765651119652261513693e-01 5.73994607496311481931172693293775e-01 1.30404180109131434539548877182824e-02
            1.56353547594157371580791959786438e-01 2.04479384117668888665164672602259e-01 6.39167068288173823020770214498043e-01 1.23034124059948190427205005903488e-02
            8.63993424651176011863640269439202e-02 2.21434582282060588021010971715441e-01 6.92166075252821810792625001340639e-01 1.02991850575796153183238956785317e-02
            3.57825581682132409788721361110220e-02 2.33702859887605390865061849581252e-01 7.30514581944181395911641629936639e-01 7.17011520488177508253491154732728e-03
            6.85809565159406497869554186763708e-03 2.40713446211374693106677113974001e-01 7.52428458137031297425778575416189e-01 3.23566049071821477262078659009603e-03
            9.93141904348405990532455689390190e-01 1.07228758486677903244421727890767e-03 5.78580806672723043510009333090238e-03 1.89313971987104344007080830580847e-05
            9.64217441831786703509976632631151e-01 5.59472991159444196784811254019587e-03 3.01878282566188545221752548286531e-02 2.25449889451542057321881129539065e-04
            9.13600657534882620858240898087388e-01 1.35088437042236320789756476301591e-02 7.28904987608937487975069302592601e-02 8.25247820126486994474213076955493e-04
            8.43646452405842683930359271471389e-01 2.44464318452784247492903801912689e-02 1.31907115748878894789797300290957e-01 1.93196977145090268562777069405456e-03
            7.57624318179077160806400570436381e-01 3.78962977032540129784798921264155e-02 2.04479384117668833154013441344432e-01 3.53471286712938540658379693581992e-03
            6.59556184463944927820477914792718e-01 5.32295983155531213237843246588454e-02 2.87214217220501943916843856641208e-01 5.49099249394542111846195808766424e-03
            5.54027474353671722795411369588692e-01 6.97293865143297497688479325006483e-02 3.76243139131998527435740697910660e-01 7.54589070896035697677550757589415e-03
            4.45972525646328055159983705380000e-01 8.66241610798276495675196429147036e-02 4.67403313273844323028072267334210e-01 9.37418906058248499468810166490584e-03
            3.40443815536054961157219622691628e-01 1.03123949278604271073689346849278e-01 5.56432235185340795524666646088008e-01 1.06379317025461911372374856910028e-02
            2.42375681820922894704750660821446e-01 1.18457249890903365541205971567251e-01 6.39167068288173823020770214498043e-01 1.10488989893644563855534812546466e-02
            1.56353547594157371580791959786438e-01 1.31907115748878950300948531548784e-01 7.11739336656963761384986355551518e-01 1.04244481108325508234013767605575e-02
            8.63993424651176011863640269439202e-02 1.42844703889933699603176364689716e-01 7.70755953644948643699308377108537e-01 8.72630427021108684426664581224031e-03
            3.57825581682132409788721361110220e-02 1.50758817682562940021284703107085e-01 8.13458624149223874510994392039720e-01 6.07510269797689825971298560602918e-03
            6.85809565159406497869554186763708e-03 1.55281260009290572599027768774249e-01 8.37860644339115445689003536244854e-01 2.74151379932027110600256669670216e-03
            9.93141904348405990532455689390190e-01 5.92534954860599906434348582706662e-04 6.26556069673340956110996202710339e-03 1.46340329542931715404779566269688e-05
            9.64217441831786703509976632631151e-01 3.09158949745345175072475107924674e-03 3.26909686707598443056177472954005e-02 1.74273513842937915552591676693339e-04
            9.13600657534882620858240898087388e-01 7.46484637840465464070360823711781e-03 7.89344960867127271031407076407049e-02 6.37919307720840266483219860305098e-04
            8.43646452405842683930359271471389e-01 1.35088437042236615692747392358797e-02 1.42844703889933644092025133431889e-01 1.49341905435466850565751073531828e-03
            7.57624318179077160806400570436381e-01 2.09410995388622962753988332451627e-02 2.21434582282060532509859740457614e-01 2.73234479413167995198730153560973e-03
            6.59556184463944927820477914792718e-01 2.94141218086309448598303362132356e-02 3.11029693727424116911350893133203e-01 4.24455544747893119550408158602295e-03
            5.54027474353671722795411369588692e-01 3.85317329733505520295011592679657e-02 4.07440792672977725175087471143343e-01 5.83299859726901532425147323124293e-03
            4.45972525646328055159983705380000e-01 4.78676094917670630346506754904112e-02 5.06159864861904895683153426944045e-01 7.24627929953755318265296025970201e-03
            3.40443815536054961157219622691628e-01 5.69852206564866667348745465915272e-02 6.02570963807458337413436311180703e-01 8.22315656190369947964313723787200e-03
            2.42375681820922894704750660821446e-01 6.54582429262553083804121456523717e-02 6.92166075252821810792625001340639e-01 8.54083564048986089434212232163190e-03
            1.56353547594157371580791959786438e-01 7.28904987608939430865362396616547e-02 7.70755953644948754721610839624191e-01 8.05813303598290273099813418866688e-03
            8.63993424651176011863640269439202e-02 7.89344960867129213921700170430995e-02 8.34666161448169519054829379456351e-01 6.74546220329458757342111496768666e-03
            3.57825581682132409788721361110220e-02 8.33077529676641559408523107777000e-02 8.80909688864122686347002399998019e-01 4.69607457652228537509087047396861e-03
            6.85809565159406497869554186763708e-03 8.58068075102570021472914163496171e-02 9.07335096838148946751800849597203e-01 2.11919927846821419537004516087109e-03
            9.93141904348405990532455689390190e-01 2.45400206576330966960070023574758e-04 6.61269544501767812111348021630874e-03 9.65314258066578720850724193436321e-06
            9.64217441831786703509976632631151e-01 1.28039146906156614535987614544865e-03 3.45021666991517303446634912234003e-02 1.14957174308260781777843440032427e-04
            9.13600657534882620858240898087388e-01 3.09158949745343917397955024739531e-03 8.33077529676639338962473857463920e-02 4.20794872583797206679750990687694e-04
            8.43646452405842683930359271471389e-01 5.59472991159443155950725667935330e-03 1.50758817682562884510133471849258e-01 9.85113748847983973846997862722219e-04
            7.57624318179077160806400570436381e-01 8.67282193331751771747661905465066e-03 2.33702859887605307598335002694512e-01 1.80235441314591259866217942686717e-03
            6.59556184463944927820477914792718e-01 1.21819506324273487546383520907511e-02 3.28261864903627742506841968861409e-01 2.79986378697032363416230538177842e-03
            5.54027474353671722795411369588692e-01 1.59580378403647096807738137158594e-02 4.30014487805963574462708720602677e-01 3.84765889950676107100768774671451e-03
            4.45972525646328055159983705380000e-01 1.98245203278485347675452743487767e-02 5.34202954025823406603024068317609e-01 4.77991046461611474027719737023290e-03
            3.40443815536054961157219622691628e-01 2.36006075357858939589572599970779e-02 6.35955576928159183047739588801051e-01 5.42429438303994315706546558431000e-03
            2.42375681820922894704750660821446e-01 2.71097362348957232613955170563713e-02 7.30514581944181395911641629936639e-01 5.63384710511353331563455526520556e-03
            1.56353547594157371580791959786438e-01 3.01878282566188094193648794316687e-02 8.13458624149223874510994392039720e-01 5.31543883858045813212145347392834e-03
            8.63993424651176011863640269439202e-02 3.26909686707597957333604199448018e-02 8.80909688864122575324699937482364e-01 4.44955321778142849631354494022162e-03
            3.57825581682132409788721361110220e-02 3.45021666991516817724061638728017e-02 9.29715275132635188271024162531830e-01 3.09770229424762661915582206972886e-03
            6.85809565159406497869554186763708e-03 3.55371579616369087650795677291171e-02 9.57604746386769067889588313846616e-01 1.39790123855750253283047879193646e-03
            9.93141904348405990532455689390190e-01 4.70334759664126607152899450792205e-05 6.81106217562759720662857176876059e-03 4.22930698511757466415573319951093e-06
            9.64217441831786703509976632631151e-01 2.45400206576335303768759965592494e-04 3.55371579616369642762307989869441e-02 5.03658965179993133704471064415742e-05
            9.13600657534882620858240898087388e-01 5.92534954860607929530424975439473e-04 8.58068075102567662248986835038522e-02 1.84361795037069428225204315019425e-04
            8.43646452405842683930359271471389e-01 1.07228758486679594599810805277684e-03 1.55281260009290517087876537516422e-01 4.31605399414993536749696190213399e-04
            7.57624318179077160806400570436381e-01 1.66223560954821786110613413001147e-03 2.40713446211374609839950267087261e-01 7.89660988168036409758088201016335e-04
            6.59556184463944927820477914792718e-01 2.33479625093991130513337850516109e-03 3.38109019285115164343835658655735e-01 1.22669725145555809885100639888833e-03
            5.54027474353671722795411369588692e-01 3.05852223886550608419887531397308e-03 4.42914003407462786299220169894397e-01 1.68576507847575832003961249228041e-03
            4.45972525646328055159983705380000e-01 3.79957341272855976185840454206755e-03 5.50227900940943426277840444527101e-01 2.09421010280397496342019714177241e-03
            3.40443815536054961157219622691628e-01 4.52329940065415454092390135087953e-03 6.55032885063290937210922493250109e-01 2.37653240194268334031790779192761e-03
            2.42375681820922894704750660821446e-01 5.19586004204584776811071122892827e-03 7.52428458137031297425778575416189e-01 2.46834320695361571565418223883626e-03
            1.56353547594157371580791959786438e-01 5.78580806672726859901656482065846e-03 8.37860644339115445689003536244854e-01 2.32883979710398793250281102018562e-03
            8.63993424651176011863640269439202e-02 6.26556069673345553128207541249139e-03 9.07335096838148946751800849597203e-01 1.94947151638543824531124570853535e-03
            3.57825581682132409788721361110220e-02 6.61269544501773016281775952052158e-03 9.57604746386769067889588313846616e-01 1.35718848462016848828437520779744e-03
            6.85809565159406497869554186763708e-03 6.81106217562765185041806503818407e-03 9.86330842172778332610505458433181e-01 6.12458940011635774233744733408003e-04
            ];
        
    case  26
        
        % ALG. DEG.:   26
        % PTS CARD.:  196
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.4e-15
        
        xyw_bar=[
            9.93141904348405990532455689390190e-01 6.81106217562754169547734051093357e-03 4.70334759664677720669700988764816e-05 4.22930698511761701580309591452789e-06
            9.64217441831786703509976632631151e-01 3.55371579616369642762307989869441e-02 2.45400206576332213792568381904857e-04 5.03658965179998215902154590217776e-05
            9.13600657534882620858240898087388e-01 8.58068075102567801026864913183090e-02 5.92534954860599039072610594303114e-04 1.84361795037071271368897540376963e-04
            8.43646452405842683930359271471389e-01 1.55281260009290517087876537516422e-01 1.07228758486679898176419101218926e-03 4.31605399414997873558386132231135e-04
            7.57624318179077160806400570436381e-01 2.40713446211374693106677113974001e-01 1.66223560954814608692231558961794e-03 7.89660988168044432854164593749147e-04
            6.59556184463944927820477914792718e-01 3.38109019285115164343835658655735e-01 2.33479625093990783568642655154690e-03 1.22669725145557045875577273363888e-03
            5.54027474353671722795411369588692e-01 4.42914003407462730788068938636570e-01 3.05852223886554641651969177473802e-03 1.68576507847577523359350326614958e-03
            4.45972525646328055159983705380000e-01 5.50227900940943426277840444527101e-01 3.79957341272851856217585009289905e-03 2.09421010280399578010190886345754e-03
            3.40443815536054961157219622691628e-01 6.55032885063290937210922493250109e-01 4.52329940065410163185788405826315e-03 2.37653240194270719276570247302516e-03
            2.42375681820922894704750660821446e-01 7.52428458137031408448081037931843e-01 5.19586004204575235831953250453807e-03 2.46834320695364043546371490833735e-03
            1.56353547594157371580791959786438e-01 8.37860644339115445689003536244854e-01 5.78580806672723824135573522653431e-03 2.32883979710401135126973670708139e-03
            8.63993424651176011863640269439202e-02 9.07335096838148946751800849597203e-01 6.26556069673345206183512345887721e-03 1.94947151638545776095035044761516e-03
            3.57825581682132409788721361110220e-02 9.57604746386769178911890776362270e-01 6.61269544501763562038831878453493e-03 1.35718848462018214923174852515331e-03
            6.85809565159406497869554186763708e-03 9.86330842172778443632807920948835e-01 6.81106217562754689964776844135486e-03 6.12458940011641954186127900783276e-04
            9.93141904348405990532455689390190e-01 6.61269544501762260996224895848172e-03 2.45400206576386857582061651328331e-04 9.65314258066566015356515378931235e-06
            9.64217441831786703509976632631151e-01 3.45021666991517303446634912234003e-02 1.28039146906156614535987614544865e-03 1.14957174308259263894801960326220e-04
            9.13600657534882620858240898087388e-01 8.33077529676639338962473857463920e-02 3.09158949745344524551171616622014e-03 4.20794872583791677248671314615081e-04
            8.43646452405842683930359271471389e-01 1.50758817682562884510133471849258e-01 5.59472991159443155950725667935330e-03 9.85113748847970963420928036669011e-04
            7.57624318179077160806400570436381e-01 2.33702859887605363109486233952339e-01 8.67282193331747608411319561128039e-03 1.80235441314588896305481924287051e-03
            6.59556184463944927820477914792718e-01 3.28261864903627686995690737603582e-01 1.21819506324273851838313476037001e-02 2.79986378697028677128844087462767e-03
            5.54027474353671722795411369588692e-01 4.30014487805963463440406258087023e-01 1.59580378403648137641823723242851e-02 3.84765889950671033034601542510700e-03
            4.45972525646328055159983705380000e-01 5.34202954025823406603024068317609e-01 1.98245203278485382369922263023909e-02 4.77991046461605229023206220517750e-03
            3.40443815536054961157219622691628e-01 6.35955576928159072025437126285397e-01 2.36006075357859668173432510229759e-02 5.42429438303987203340295053521913e-03
            2.42375681820922894704750660821446e-01 7.30514581944181395911641629936639e-01 2.71097362348957648947589404997416e-02 5.63384710511345872252508826250050e-03
            1.56353547594157371580791959786438e-01 8.13458624149223874510994392039720e-01 3.01878282566188094193648794316687e-02 5.31543883858038787582067641324102e-03
            8.63993424651176011863640269439202e-02 8.80909688864122575324699937482364e-01 3.26909686707598234889360355737153e-02 4.44955321778136951571536172878041e-03
            3.57825581682132409788721361110220e-02 9.29715275132635077248721700016176e-01 3.45021666991517372835573951306287e-02 3.09770229424758585315413661476214e-03
            6.85809565159406497869554186763708e-03 9.57604746386769067889588313846616e-01 3.55371579616369226428673755435739e-02 1.39790123855748410139354653836108e-03
            9.93141904348405990532455689390190e-01 6.26556069673335578468220674608347e-03 5.92534954860653682862103863726588e-04 1.46340329542932308327842644279926e-05
            9.64217441831786703509976632631151e-01 3.26909686707598512445116512026289e-02 3.09158949745344524551171616622014e-03 1.74273513842938620284003792271221e-04
            9.13600657534882620858240898087388e-01 7.89344960867127409809285154551617e-02 7.46484637840463816083058645745041e-03 6.37919307720842868568433825515740e-04
            8.43646452405842683930359271471389e-01 1.42844703889933699603176364689716e-01 1.35088437042236164664643638388952e-02 1.49341905435467457718967665414311e-03
            7.57624318179077160806400570436381e-01 2.21434582282060643532162202973268e-01 2.09410995388621956614372265903512e-02 2.73234479413169122768989538485584e-03
            6.59556184463944927820477914792718e-01 3.11029693727424227933653355648858e-01 2.94141218086308442458687295584241e-02 4.24455544747894854273884135409389e-03
            5.54027474353671722795411369588692e-01 4.07440792672977780686238702401170e-01 3.85317329733504965183499280101387e-02 5.83299859726903874301839891813870e-03
            4.45972525646328055159983705380000e-01 5.06159864861905006705455889459699e-01 4.78676094917669381345604051603004e-02 7.24627929953758267295205186542262e-03
            3.40443815536054961157219622691628e-01 6.02570963807458559458041236212011e-01 5.69852206564864793847391410963610e-02 8.22315656190373243938918079720679e-03
            2.42375681820922894704750660821446e-01 6.92166075252822032837229926371947e-01 6.54582429262551279691706440644339e-02 8.54083564048989558881164185777379e-03
            1.56353547594157371580791959786438e-01 7.70755953644948976766215764655499e-01 7.28904987608937071641435068158899e-02 8.05813303598293569074417774800168e-03
            8.63993424651176011863640269439202e-02 8.34666161448169630077131841972005e-01 7.89344960867127687365041310840752e-02 6.74546220329461532899673059660017e-03
            3.57825581682132409788721361110220e-02 8.80909688864122908391607325029327e-01 8.33077529676639061406717701174784e-02 4.69607457652230445704910621884665e-03
            6.85809565159406497869554186763708e-03 9.07335096838149168796405774628511e-01 8.58068075102568217360499147616792e-02 2.11919927846822286898742504490656e-03
            9.93141904348405990532455689390190e-01 5.78580806672717492394886207307536e-03 1.07228758486683454359544853673469e-03 1.89313971987104344007080830580847e-05
            9.64217441831786703509976632631151e-01 3.01878282566188545221752548286531e-02 5.59472991159444196784811254019587e-03 2.25449889451542057321881129539065e-04
            9.13600657534882620858240898087388e-01 7.28904987608937487975069302592601e-02 1.35088437042236303442521716533520e-02 8.25247820126486994474213076955493e-04
            8.43646452405842683930359271471389e-01 1.31907115748878894789797300290957e-01 2.44464318452784212798434282376547e-02 1.93196977145090268562777069405456e-03
            7.57624318179077160806400570436381e-01 2.04479384117668888665164672602259e-01 3.78962977032539505284347569613601e-02 3.53471286712938540658379693581992e-03
            6.59556184463944927820477914792718e-01 2.87214217220501943916843856641208e-01 5.32295983155531282626782285660738e-02 5.49099249394542111846195808766424e-03
            5.54027474353671722795411369588692e-01 3.76243139131998471924589466652833e-01 6.97293865143298052799991637584753e-02 7.54589070896035697677550757589415e-03
            4.45972525646328055159983705380000e-01 4.67403313273844323028072267334210e-01 8.66241610798276218119440272857901e-02 9.37418906058248499468810166490584e-03
            3.40443815536054961157219622691628e-01 5.56432235185340795524666646088008e-01 1.03123949278604243318113731220365e-01 1.06379317025461911372374856910028e-02
            2.42375681820922894704750660821446e-01 6.39167068288173823020770214498043e-01 1.18457249890903337785630355938338e-01 1.10488989893644563855534812546466e-02
            1.56353547594157371580791959786438e-01 7.11739336656963761384986355551518e-01 1.31907115748878922545372915919870e-01 1.04244481108325508234013767605575e-02
            8.63993424651176011863640269439202e-02 7.70755953644948754721610839624191e-01 1.42844703889933644092025133431889e-01 8.72630427021108684426664581224031e-03
            3.57825581682132409788721361110220e-02 8.13458624149223874510994392039720e-01 1.50758817682562940021284703107085e-01 6.07510269797689825971298560602918e-03
            6.85809565159406497869554186763708e-03 8.37860644339115445689003536244854e-01 1.55281260009290544843452153145336e-01 2.74151379932027110600256669670216e-03
            9.93141904348405990532455689390190e-01 5.19586004204576363402212635378419e-03 1.66223560954824583352218425602587e-03 2.23437044034388537338663843767605e-05
            9.64217441831786703509976632631151e-01 2.71097362348957683642058924533558e-02 8.67282193331752812581747491549322e-03 2.66086313377670777551714520470227e-04
            9.13600657534882620858240898087388e-01 6.54582429262551418469584518788906e-02 2.09410995388622372948006500337215e-02 9.73995377041929756394000872177230e-04
            8.43646452405842683930359271471389e-01 1.18457249890903323907842548123881e-01 3.78962977032539921617981804047304e-02 2.28019945049902361677851558852126e-03
            7.57624318179077160806400570436381e-01 1.83629710682765651119652261513693e-01 5.87459711381571880739471680499264e-02 4.17183045842758372379055487044752e-03
            6.59556184463944927820477914792718e-01 2.57928513623787214470439721480943e-01 8.25153019122678577090823637263384e-02 6.48072151666520025747875166644008e-03
            5.54027474353671722795411369588692e-01 3.37879630669400410347691376955481e-01 1.08092894976927866856897253455827e-01 8.90600676178041061881174300651764e-03
            4.45972525646328055159983705380000e-01 4.19744687509676861481011655996554e-01 1.34282786843995083359004638623446e-01 1.10638484414596078436732184968605e-02
            3.40443815536054961157219622691628e-01 4.99695804555290001847112080213265e-01 1.59860379908655036995668297095108e-01 1.25553755452267462588977764426090e-02
            2.42375681820922894704750660821446e-01 5.73994607496311592953475155809429e-01 1.83629710682765567852925414626952e-01 1.30404180109131191678262240429831e-02
            1.56353547594157371580791959786438e-01 6.39167068288173823020770214498043e-01 2.04479384117668860909589056973346e-01 1.23034124059947964913153128918566e-02
            8.63993424651176011863640269439202e-02 6.92166075252821810792625001340639e-01 2.21434582282060588021010971715441e-01 1.02991850575795962363656599336537e-02
            3.57825581682132409788721361110220e-02 7.30514581944181506933944092452293e-01 2.33702859887605307598335002694512e-01 7.17011520488176207210884172127408e-03
            6.85809565159406497869554186763708e-03 7.52428458137031408448081037931843e-01 2.40713446211374582084374651458347e-01 3.23566049071820870108862067127120e-03
            9.93141904348405990532455689390190e-01 4.52329940065408081517617233657802e-03 2.33479625093992865236813827323203e-03 2.47112936176651894551459243132285e-05
            9.64217441831786703509976632631151e-01 2.36006075357859286534267795332198e-02 1.21819506324273678365965878356292e-02 2.94281418102975984418118171603851e-04
            9.13600657534882620858240898087388e-01 5.69852206564865071403147567252745e-02 2.94141218086308720014443451873376e-02 1.07720211965511391931371587560307e-03
            8.43646452405842683930359271471389e-01 1.03123949278604215562538115591451e-01 5.32295983155531005071026129371603e-02 2.52181451699871406538222551318995e-03
            7.57624318179077160806400570436381e-01 1.59860379908655036995668297095108e-01 8.25153019122678021979311324685114e-02 4.61388700458534046361380731582358e-03
            6.59556184463944927820477914792718e-01 2.24541823999307582626627777244721e-01 1.15901991536747489552894307962561e-01 7.16743335666338628880955852196166e-03
            5.54027474353671722795411369588692e-01 2.94143937391041054851825720106717e-01 1.51828588255287222352762910304591e-01 9.84970728566367768885836397885214e-03
            4.45972525646328055159983705380000e-01 3.65412247072903983990954657201655e-01 1.88615227280767960849061637418345e-01 1.22361987270193754706593125547442e-02
            3.40443815536054961157219622691628e-01 4.35014360464637428460576984434738e-01 2.24541823999307610382203392873635e-01 1.38857714001264769682064681433076e-02
            2.42375681820922894704750660821446e-01 4.99695804555289946335960848955438e-01 2.57928513623787214470439721480943e-01 1.44222100573066867174043537147554e-02
            1.56353547594157371580791959786438e-01 5.56432235185340684502364183572354e-01 2.87214217220501999427995087899035e-01 1.36071096794929437268084626566633e-02
            8.63993424651176011863640269439202e-02 6.02570963807458226391133848665049e-01 3.11029693727424172422502124391031e-01 1.13905098897275538355478019525435e-02
            3.57825581682132409788721361110220e-02 6.35955576928159072025437126285397e-01 3.28261864903627742506841968861409e-01 7.92987675190731135244082139479360e-03
            6.85809565159406497869554186763708e-03 6.55032885063290826188620030734455e-01 3.38109019285115164343835658655735e-01 3.57851835978058698331527587299661e-03
            9.93141904348405990532455689390190e-01 3.79957341272849687813240038281037e-03 3.05852223886551258941191022699968e-03 2.59234313538678390104099541479954e-05
            9.64217441831786703509976632631151e-01 1.98245203278485590536739380240761e-02 1.59580378403647374363494293447729e-02 3.08716502622017469059656491481292e-04
            9.13600657534882620858240898087388e-01 4.78676094917669173178786934386153e-02 3.85317329733504618238804084739968e-02 1.13004101020263434188017903636592e-03
            8.43646452405842683930359271471389e-01 8.66241610798275801785806038424198e-02 6.97293865143297358910601246861916e-02 2.64551449754415286472197799128026e-03
            7.57624318179077160806400570436381e-01 1.34282786843995055603429022994533e-01 1.08092894976927783590170406569086e-01 4.84020727075039954717672330275491e-03
            6.59556184463944927820477914792718e-01 1.88615227280767905337910406160518e-01 1.51828588255287166841611679046764e-01 7.51901011252851213251258144509848e-03
            5.54027474353671722795411369588692e-01 2.47081032014963314358269030890369e-01 1.98891493631364962846319599520939e-01 1.03328548730068289773775802586897e-02
            4.45972525646328055159983705380000e-01 3.06946442338708491703869185585063e-01 2.47081032014963453136147109034937e-01 1.28364084308971292819334664159214e-02
            3.40443815536054961157219622691628e-01 3.65412247072903872968652194686001e-01 2.94143937391041165874128182622371e-01 1.45668959001544577602782482017574e-02
            2.42375681820922894704750660821446e-01 4.19744687509676694947557962223073e-01 3.37879630669400465858842608213308e-01 1.51296479324896284562873560730623e-02
            1.56353547594157371580791959786438e-01 4.67403313273844156494618573560729e-01 3.76243139131998527435740697910660e-01 1.42745652719986730605494784640541e-02
            8.63993424651176011863640269439202e-02 5.06159864861904673638548501912737e-01 4.07440792672977725175087471143343e-01 1.19492368866039224556407916111311e-02
            3.57825581682132409788721361110220e-02 5.34202954025823184558419143286301e-01 4.30014487805963629973859951860504e-01 8.31885286150084793388437276462355e-03
            6.85809565159406497869554186763708e-03 5.50227900940943204233235519495793e-01 4.42914003407462786299220169894397e-01 3.75405174992586186319409868872299e-03
            9.93141904348405990532455689390190e-01 3.05852223886545534353720299236556e-03 3.79957341272855412400710761744449e-03 2.59234313538678762798596333372103e-05
            9.64217441831786703509976632631151e-01 1.59580378403647304974555254375446e-02 1.98245203278485659925678419313044e-02 3.08716502622017902740525485683065e-04
            9.13600657534882620858240898087388e-01 3.85317329733504410071986967523117e-02 4.78676094917669381345604051603004e-02 1.13004101020263585976322051607212e-03
            8.43646452405842683930359271471389e-01 6.97293865143296942576967012428213e-02 8.66241610798276218119440272857901e-02 2.64551449754415633416892994489444e-03
            7.57624318179077160806400570436381e-01 1.08092894976927783590170406569086e-01 1.34282786843995055603429022994533e-01 4.84020727075040648607062720998329e-03
            6.59556184463944927820477914792718e-01 1.51828588255287111330460447788937e-01 1.88615227280767960849061637418345e-01 7.51901011252852340821517529434459e-03
            5.54027474353671722795411369588692e-01 1.98891493631364796312865905747458e-01 2.47081032014963480891722724663850e-01 1.03328548730068445898888640499536e-02
            4.45972525646328055159983705380000e-01 2.47081032014963314358269030890369e-01 3.06946442338708602726171648100717e-01 1.28364084308971483638917021607995e-02
            3.40443815536054961157219622691628e-01 2.94143937391040999340674488848890e-01 3.65412247072904039502105888459482e-01 1.45668959001544785769599599234425e-02
            2.42375681820922894704750660821446e-01 3.37879630669400299325388914439827e-01 4.19744687509676861481011655996554e-01 1.51296479324896492729690677947474e-02
            1.56353547594157371580791959786438e-01 3.76243139131998360902287004137179e-01 4.67403313273844323028072267334210e-01 1.42745652719986921425077142089322e-02
            8.63993424651176011863640269439202e-02 4.07440792672977503130482546112034e-01 5.06159864861904895683153426944045e-01 1.19492368866039398028755513792021e-02
            3.57825581682132409788721361110220e-02 4.30014487805963352418103795571369e-01 5.34202954025823517625326530833263e-01 8.31885286150086007694870460227321e-03
            6.85809565159406497869554186763708e-03 4.42914003407462564254615244863089e-01 5.50227900940943426277840444527101e-01 3.75405174992586706736452661914427e-03
            9.93141904348405990532455689390190e-01 2.33479625093987270753603802120324e-03 4.52329940065413632632740359440504e-03 2.47112936176650776467968867455838e-05
            9.64217441831786703509976632631151e-01 1.21819506324273643671496358820150e-02 2.36006075357859321228737314868340e-02 2.94281418102974683375511188998530e-04
            9.13600657534882620858240898087388e-01 2.94141218086308615931034893264950e-02 5.69852206564865210181025645397312e-02 1.07720211965510914882415693938356e-03
            8.43646452405842683930359271471389e-01 5.32295983155530866293148051227035e-02 1.03123949278604229440325923405908e-01 2.52181451699870278967963166394384e-03
            7.57624318179077160806400570436381e-01 8.25153019122678299535067480974249e-02 1.59860379908655009240092681466194e-01 4.61388700458532051429383358254199e-03
            6.59556184463944927820477914792718e-01 1.15901991536747461797318692333647e-01 2.24541823999307610382203392873635e-01 7.16743335666335419642525295103042e-03
            5.54027474353671722795411369588692e-01 1.51828588255287111330460447788937e-01 2.94143937391041165874128182622371e-01 9.84970728566363432077146455867478e-03
            4.45972525646328055159983705380000e-01 1.88615227280767905337910406160518e-01 3.65412247072904039502105888459482e-01 1.22361987270193216942315572737243e-02
            3.40443815536054961157219622691628e-01 2.24541823999307554871052161615808e-01 4.35014360464637483971728215692565e-01 1.38857714001264145181613329782522e-02
            2.42375681820922894704750660821446e-01 2.57928513623787158959288490223116e-01 4.99695804555290001847112080213265e-01 1.44222100573066225326357425728929e-02
            1.56353547594157371580791959786438e-01 2.87214217220501888405692625383381e-01 5.56432235185340795524666646088008e-01 1.36071096794928830114868034684150e-02
            8.63993424651176011863640269439202e-02 3.11029693727424005889048430617549e-01 6.02570963807458337413436311180703e-01 1.13905098897275017938435226483307e-02
            3.57825581682132409788721361110220e-02 3.28261864903627631484539506345754e-01 6.35955576928159183047739588801051e-01 7.92987675190727665797130185865171e-03
            6.85809565159406497869554186763708e-03 3.38109019285115053321533196140081e-01 6.55032885063290937210922493250109e-01 3.57851835978057093712312308753098e-03
            9.93141904348405990532455689390190e-01 1.66223560954819097289225648950151e-03 5.19586004204581827781161962320766e-03 2.23437044034388943914478525831768e-05
            9.64217441831786703509976632631151e-01 8.67282193331753159526442686910741e-03 2.71097362348957648947589404997416e-02 2.66086313377671265442692138947223e-04
            9.13600657534882620858240898087388e-01 2.09410995388622407642476019873357e-02 6.54582429262551418469584518788906e-02 9.73995377041931599537694097534768e-04
            8.43646452405842683930359271471389e-01 3.78962977032539991006920843119588e-02 1.18457249890903310030054740309424e-01 2.28019945049902795358720553053899e-03
            7.57624318179077160806400570436381e-01 5.87459711381572505239923032149818e-02 1.83629710682765595608501030255866e-01 4.17183045842759153004619676607945e-03
            6.59556184463944927820477914792718e-01 8.25153019122678438312945559118816e-02 2.57928513623787214470439721480943e-01 6.48072151666521153318134551568619e-03
            5.54027474353671722795411369588692e-01 1.08092894976927825223533830012457e-01 3.37879630669400465858842608213308e-01 8.90600676178042796604650277458859e-03
            4.45972525646328055159983705380000e-01 1.34282786843995111114580254252360e-01 4.19744687509676861481011655996554e-01 1.10638484414596269256314542417385e-02
            3.40443815536054961157219622691628e-01 1.59860379908655064751243912724021e-01 4.99695804555289946335960848955438e-01 1.25553755452267688103029641411013e-02
            2.42375681820922894704750660821446e-01 1.83629710682765651119652261513693e-01 5.73994607496311481931172693293775e-01 1.30404180109131434539548877182824e-02
            1.56353547594157371580791959786438e-01 2.04479384117668888665164672602259e-01 6.39167068288173823020770214498043e-01 1.23034124059948190427205005903488e-02
            8.63993424651176011863640269439202e-02 2.21434582282060588021010971715441e-01 6.92166075252821810792625001340639e-01 1.02991850575796153183238956785317e-02
            3.57825581682132409788721361110220e-02 2.33702859887605390865061849581252e-01 7.30514581944181395911641629936639e-01 7.17011520488177508253491154732728e-03
            6.85809565159406497869554186763708e-03 2.40713446211374693106677113974001e-01 7.52428458137031297425778575416189e-01 3.23566049071821477262078659009603e-03
            9.93141904348405990532455689390190e-01 1.07228758486677903244421727890767e-03 5.78580806672723043510009333090238e-03 1.89313971987104344007080830580847e-05
            9.64217441831786703509976632631151e-01 5.59472991159444196784811254019587e-03 3.01878282566188545221752548286531e-02 2.25449889451542057321881129539065e-04
            9.13600657534882620858240898087388e-01 1.35088437042236320789756476301591e-02 7.28904987608937487975069302592601e-02 8.25247820126486994474213076955493e-04
            8.43646452405842683930359271471389e-01 2.44464318452784247492903801912689e-02 1.31907115748878894789797300290957e-01 1.93196977145090268562777069405456e-03
            7.57624318179077160806400570436381e-01 3.78962977032540129784798921264155e-02 2.04479384117668833154013441344432e-01 3.53471286712938540658379693581992e-03
            6.59556184463944927820477914792718e-01 5.32295983155531213237843246588454e-02 2.87214217220501943916843856641208e-01 5.49099249394542111846195808766424e-03
            5.54027474353671722795411369588692e-01 6.97293865143297497688479325006483e-02 3.76243139131998527435740697910660e-01 7.54589070896035697677550757589415e-03
            4.45972525646328055159983705380000e-01 8.66241610798276495675196429147036e-02 4.67403313273844323028072267334210e-01 9.37418906058248499468810166490584e-03
            3.40443815536054961157219622691628e-01 1.03123949278604271073689346849278e-01 5.56432235185340795524666646088008e-01 1.06379317025461911372374856910028e-02
            2.42375681820922894704750660821446e-01 1.18457249890903365541205971567251e-01 6.39167068288173823020770214498043e-01 1.10488989893644563855534812546466e-02
            1.56353547594157371580791959786438e-01 1.31907115748878950300948531548784e-01 7.11739336656963761384986355551518e-01 1.04244481108325508234013767605575e-02
            8.63993424651176011863640269439202e-02 1.42844703889933699603176364689716e-01 7.70755953644948643699308377108537e-01 8.72630427021108684426664581224031e-03
            3.57825581682132409788721361110220e-02 1.50758817682562940021284703107085e-01 8.13458624149223874510994392039720e-01 6.07510269797689825971298560602918e-03
            6.85809565159406497869554186763708e-03 1.55281260009290572599027768774249e-01 8.37860644339115445689003536244854e-01 2.74151379932027110600256669670216e-03
            9.93141904348405990532455689390190e-01 5.92534954860599906434348582706662e-04 6.26556069673340956110996202710339e-03 1.46340329542931715404779566269688e-05
            9.64217441831786703509976632631151e-01 3.09158949745345175072475107924674e-03 3.26909686707598443056177472954005e-02 1.74273513842937915552591676693339e-04
            9.13600657534882620858240898087388e-01 7.46484637840465464070360823711781e-03 7.89344960867127271031407076407049e-02 6.37919307720840266483219860305098e-04
            8.43646452405842683930359271471389e-01 1.35088437042236615692747392358797e-02 1.42844703889933644092025133431889e-01 1.49341905435466850565751073531828e-03
            7.57624318179077160806400570436381e-01 2.09410995388622962753988332451627e-02 2.21434582282060532509859740457614e-01 2.73234479413167995198730153560973e-03
            6.59556184463944927820477914792718e-01 2.94141218086309448598303362132356e-02 3.11029693727424116911350893133203e-01 4.24455544747893119550408158602295e-03
            5.54027474353671722795411369588692e-01 3.85317329733505520295011592679657e-02 4.07440792672977725175087471143343e-01 5.83299859726901532425147323124293e-03
            4.45972525646328055159983705380000e-01 4.78676094917670630346506754904112e-02 5.06159864861904895683153426944045e-01 7.24627929953755318265296025970201e-03
            3.40443815536054961157219622691628e-01 5.69852206564866667348745465915272e-02 6.02570963807458337413436311180703e-01 8.22315656190369947964313723787200e-03
            2.42375681820922894704750660821446e-01 6.54582429262553083804121456523717e-02 6.92166075252821810792625001340639e-01 8.54083564048986089434212232163190e-03
            1.56353547594157371580791959786438e-01 7.28904987608939430865362396616547e-02 7.70755953644948754721610839624191e-01 8.05813303598290273099813418866688e-03
            8.63993424651176011863640269439202e-02 7.89344960867129213921700170430995e-02 8.34666161448169519054829379456351e-01 6.74546220329458757342111496768666e-03
            3.57825581682132409788721361110220e-02 8.33077529676641559408523107777000e-02 8.80909688864122686347002399998019e-01 4.69607457652228537509087047396861e-03
            6.85809565159406497869554186763708e-03 8.58068075102570021472914163496171e-02 9.07335096838148946751800849597203e-01 2.11919927846821419537004516087109e-03
            9.93141904348405990532455689390190e-01 2.45400206576330966960070023574758e-04 6.61269544501767812111348021630874e-03 9.65314258066578720850724193436321e-06
            9.64217441831786703509976632631151e-01 1.28039146906156614535987614544865e-03 3.45021666991517303446634912234003e-02 1.14957174308260781777843440032427e-04
            9.13600657534882620858240898087388e-01 3.09158949745343917397955024739531e-03 8.33077529676639338962473857463920e-02 4.20794872583797206679750990687694e-04
            8.43646452405842683930359271471389e-01 5.59472991159443155950725667935330e-03 1.50758817682562884510133471849258e-01 9.85113748847983973846997862722219e-04
            7.57624318179077160806400570436381e-01 8.67282193331751771747661905465066e-03 2.33702859887605307598335002694512e-01 1.80235441314591259866217942686717e-03
            6.59556184463944927820477914792718e-01 1.21819506324273487546383520907511e-02 3.28261864903627742506841968861409e-01 2.79986378697032363416230538177842e-03
            5.54027474353671722795411369588692e-01 1.59580378403647096807738137158594e-02 4.30014487805963574462708720602677e-01 3.84765889950676107100768774671451e-03
            4.45972525646328055159983705380000e-01 1.98245203278485347675452743487767e-02 5.34202954025823406603024068317609e-01 4.77991046461611474027719737023290e-03
            3.40443815536054961157219622691628e-01 2.36006075357858939589572599970779e-02 6.35955576928159183047739588801051e-01 5.42429438303994315706546558431000e-03
            2.42375681820922894704750660821446e-01 2.71097362348957232613955170563713e-02 7.30514581944181395911641629936639e-01 5.63384710511353331563455526520556e-03
            1.56353547594157371580791959786438e-01 3.01878282566188094193648794316687e-02 8.13458624149223874510994392039720e-01 5.31543883858045813212145347392834e-03
            8.63993424651176011863640269439202e-02 3.26909686707597957333604199448018e-02 8.80909688864122575324699937482364e-01 4.44955321778142849631354494022162e-03
            3.57825581682132409788721361110220e-02 3.45021666991516817724061638728017e-02 9.29715275132635188271024162531830e-01 3.09770229424762661915582206972886e-03
            6.85809565159406497869554186763708e-03 3.55371579616369087650795677291171e-02 9.57604746386769067889588313846616e-01 1.39790123855750253283047879193646e-03
            9.93141904348405990532455689390190e-01 4.70334759664126607152899450792205e-05 6.81106217562759720662857176876059e-03 4.22930698511757466415573319951093e-06
            9.64217441831786703509976632631151e-01 2.45400206576335303768759965592494e-04 3.55371579616369642762307989869441e-02 5.03658965179993133704471064415742e-05
            9.13600657534882620858240898087388e-01 5.92534954860607929530424975439473e-04 8.58068075102567662248986835038522e-02 1.84361795037069428225204315019425e-04
            8.43646452405842683930359271471389e-01 1.07228758486679594599810805277684e-03 1.55281260009290517087876537516422e-01 4.31605399414993536749696190213399e-04
            7.57624318179077160806400570436381e-01 1.66223560954821786110613413001147e-03 2.40713446211374609839950267087261e-01 7.89660988168036409758088201016335e-04
            6.59556184463944927820477914792718e-01 2.33479625093991130513337850516109e-03 3.38109019285115164343835658655735e-01 1.22669725145555809885100639888833e-03
            5.54027474353671722795411369588692e-01 3.05852223886550608419887531397308e-03 4.42914003407462786299220169894397e-01 1.68576507847575832003961249228041e-03
            4.45972525646328055159983705380000e-01 3.79957341272855976185840454206755e-03 5.50227900940943426277840444527101e-01 2.09421010280397496342019714177241e-03
            3.40443815536054961157219622691628e-01 4.52329940065415454092390135087953e-03 6.55032885063290937210922493250109e-01 2.37653240194268334031790779192761e-03
            2.42375681820922894704750660821446e-01 5.19586004204584776811071122892827e-03 7.52428458137031297425778575416189e-01 2.46834320695361571565418223883626e-03
            1.56353547594157371580791959786438e-01 5.78580806672726859901656482065846e-03 8.37860644339115445689003536244854e-01 2.32883979710398793250281102018562e-03
            8.63993424651176011863640269439202e-02 6.26556069673345553128207541249139e-03 9.07335096838148946751800849597203e-01 1.94947151638543824531124570853535e-03
            3.57825581682132409788721361110220e-02 6.61269544501773016281775952052158e-03 9.57604746386769067889588313846616e-01 1.35718848462016848828437520779744e-03
            6.85809565159406497869554186763708e-03 6.81106217562765185041806503818407e-03 9.86330842172778332610505458433181e-01 6.12458940011635774233744733408003e-04
            ];
        
    case  27
        
        % ALG. DEG.:   27
        % PTS CARD.:  225
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.3e-15
        
        xyw_bar=[
            9.93996259010242688702874147566035e-01 5.96769608388527435932635611948172e-03 3.60449058720369377994963144828944e-05 2.83905472259312966406825766430000e-06
            9.68636696200352975694158885744400e-01 3.11750066470509283600431871263936e-02 1.88297152596095945797927129206073e-04 3.39348451645763828977454823476734e-05
            9.24103291705213436557642125990242e-01 7.54410441162094991263131760206306e-02 4.55664178577064316044697989127599e-04 1.25058552118981712846651488391103e-04
            8.62208865680085256855136321973987e-01 1.36963872038773165629166328471911e-01 8.27262281141577515697349554102402e-04 2.95717055589187361591685387907091e-04
            7.85486086304269637281549876206554e-01 2.13226027719202099186546206510684e-01 1.28788597652826353190391728276154e-03 5.48438834010235272756195179510996e-04
            6.97075673538781748206361044140067e-01 3.01105647265648423971384772812598e-01 1.81867919556982782225418304733466e-03 8.67129106424570773696547387743294e-04
            6.00597046998717187804572859022301e-01 3.97005041120918888974955507364939e-01 2.39791188036392322047163361276034e-03 1.21866058435084385382085603310998e-03
            5.00000000000000000000000000000000e-01 4.96998129505121288840285842525191e-01 3.00187049487871115971415747480933e-03 1.55748442427121325248862149237539e-03
            3.99402953001282645661973447204218e-01 5.96991217889323966261372333974577e-01 3.60582910939344358780545007903129e-03 1.83254515960601968398158145845400e-03
            3.02924326461218529349395112149068e-01 6.92890611744594098198035680979956e-01 4.18506179418742796372043812880293e-03 1.99539803543435229071789649424318e-03
            2.14513913695730473740752586309100e-01 7.80770231291040506249601094168611e-01 4.71585501322902000964631952228956e-03 2.00821972748592121951749511765684e-03
            1.37791134319915020700619834315148e-01 8.57032386971469328784678509691730e-01 5.17647870861570602585288725094870e-03 1.85040836132341563260994554696026e-03
            7.58967082947862858866017177206231e-02 9.18555214894033356110014665318886e-01 5.54807681118030249223238570266403e-03 1.52268816745214168560662582052601e-03
            3.13633037996470243058411142555997e-02 9.62821252363191826262323047558311e-01 5.81544383716114943183583818608895e-03 1.04805719819146348284477632262224e-03
            6.00374098975725578597462117613759e-03 9.88028562926357434292867765179835e-01 5.96769608388525441000638238620013e-03 4.70041891913291955823472889619552e-04
            9.93996259010242688702874147566035e-01 5.81544383716121274924271133954790e-03 1.88297152596098547883141094416715e-04 6.49599998128798524453519913635091e-06
            9.68636696200352975694158885744400e-01 3.03796469744181257821580288691621e-02 9.83656825228898523683085386437597e-04 7.76458276058705732739634308714471e-05
            9.24103291705213436557642125990242e-01 7.35163367751439783548406126101327e-02 2.38037151964258508751726139962557e-03 2.86144661376163094834018663092934e-04
            8.62208865680085256855136321973987e-01 1.33469549113341329205084662135050e-01 4.32158520657341393977901589096291e-03 6.76625910831095368422327673840755e-04
            7.85486086304269637281549876206554e-01 2.07786048651239951823299634270370e-01 6.72786504449041089515048952307552e-03 1.25487495084773800942123944679452e-03
            6.97075673538781748206361044140067e-01 2.93423618782111594871508941650973e-01 9.50070767910665692213001420896035e-03 1.98406554628272188342541504368910e-03
            6.00597046998717187804572859022301e-01 3.86876356847827451712618085366557e-01 1.25265961534553604828090556111420e-02 2.78839962827796611008479565896323e-03
            5.00000000000000000000000000000000e-01 4.84318348100176432335928211614373e-01 1.56816518998235676640717883856269e-02 3.56365754784786345141234242817063e-03
            3.99402953001282645661973447204218e-01 5.81760339352525801537296956666978e-01 1.88367076461916083118808273866307e-02 4.19302003155368717413065127175287e-03
            3.02924326461218529349395112149068e-01 6.75213077418241103266893787804293e-01 2.18625961205404228948623313044664e-02 4.56564133748162907094902962512606e-03
            2.14513913695730473740752586309100e-01 7.60850647549112912848556788958376e-01 2.46354387551566134106906247325242e-02 4.59497846531655828455908974206068e-03
            1.37791134319915020700619834315148e-01 8.35167147087011452200044914206956e-01 2.70417185930735826104864827357233e-02 4.23389256461848169366124494672476e-03
            7.58967082947862858866017177206231e-02 8.95120359425209177750559774722205e-01 2.89829322800044808516872762993444e-02 3.48404073671464099490324706209776e-03
            3.13633037996470243058411142555997e-02 9.38257049225934891545364280318609e-01 3.03796469744180841487946054257918e-02 2.39804449194344669951672877061810e-03
            6.00374098975725578597462117613759e-03 9.62821252363191826262323047558311e-01 3.11750066470508624405511000077240e-02 1.07549604337475149425096265787261e-03
            9.93996259010242688702874147566035e-01 5.54807681118029902278543374904984e-03 4.55664178577012274340418684914766e-04 9.89264452102744639154681405290859e-06
            9.68636696200352975694158885744400e-01 2.89829322800044947294750841138011e-02 2.38037151964252957636603014179855e-03 1.18245470021315912125502156637680e-04
            9.24103291705213436557642125990242e-01 7.01363979648026436564123287098482e-02 5.76031032998391978594554529991001e-03 4.35764689768809829369111685792859e-04
            8.62208865680085256855136321973987e-01 1.27333240792828483733956090873107e-01 1.04578935270862594109075871529058e-02 1.03042174089435162667727219343305e-03
            7.85486086304269637281549876206554e-01 1.98233013762792537049151064820762e-01 1.62808999329378256692990589726833e-02 1.91102706940234436973347076360596e-03
            6.97075673538781748206361044140067e-01 2.79933367220396467356380298951990e-01 2.29909592408217844372586569079431e-02 3.02149864721850671886449468672708e-03
            6.00597046998717187804572859022301e-01 3.69089583585268110610400071891490e-01 3.03133694160147015850270690862089e-02 4.24640492373426405720016063582989e-03
            5.00000000000000000000000000000000e-01 4.62051645852606662767669831737294e-01 3.79483541473933372323301682627061e-02 5.42703162208835708513188222923418e-03
            3.99402953001282645661973447204218e-01 5.55013708119945547991846979130059e-01 4.55833388787718618573308049235493e-02 6.38547671816395469301941290041214e-03
            3.02924326461218529349395112149068e-01 6.44169924484816802667808133264771e-01 5.29057490539647234939479858439881e-02 6.95293517430995832578899396025918e-03
            2.14513913695730473740752586309100e-01 7.25870277942420871752915445540566e-01 5.96158083618486545063319681503344e-02 6.99761217212013179184149080924726e-03
            1.37791134319915020700619834315148e-01 7.96770050912384841801383572601480e-01 6.54388147677001930091478243411984e-02 6.44772078242637249057001014307389e-03
            7.58967082947862858866017177206231e-02 8.53966893740411014945834722311702e-01 7.01363979648026436564123287098482e-02 5.30578457579717385067530344144870e-03
            3.13633037996470243058411142555997e-02 8.95120359425208955705954849690897e-01 7.35163367751440199882040360535029e-02 3.65193992806951001073301199539856e-03
            6.00374098975725578597462117613759e-03 9.18555214894033134065409740287578e-01 7.54410441162095546374644072784577e-02 1.63785407505010480525009519681134e-03
            9.93996259010242688702874147566035e-01 5.17647870861573117934328891465157e-03 8.27262281141580117782563519313044e-04 1.28847811346783016906705418569423e-05
            9.68636696200352975694158885744400e-01 2.70417185930736173049560022718651e-02 4.32158520657340700088511198373453e-03 1.54010082759304777876713243855988e-04
            9.24103291705213436557642125990242e-01 6.54388147677002346425112477845687e-02 1.04578935270863287998466262251895e-02 5.67566401679311591244037060732808e-04
            8.62208865680085256855136321973987e-01 1.18804737622746001135531912495935e-01 1.89863966971687420093317655300780e-02 1.34208386641381069610823306703651e-03
            7.85486086304269637281549876206554e-01 1.84955798200191423585891925540636e-01 2.95581154955389391325581982528092e-02 2.48903773701327109268977011424795e-03
            6.97075673538781748206361044140067e-01 2.61184039905030807027230821404373e-01 4.17402865561874447664081344555598e-02 3.93538337351412650932935832770454e-03
            6.00597046998717187804572859022301e-01 3.44368767056512481428143246375839e-01 5.50341859447703307672838946018601e-02 5.53077571272658340822969691430444e-03
            5.00000000000000000000000000000000e-01 4.31104432840042572916416929729166e-01 6.88955671599574270835830702708336e-02 7.06849563965977322615996669696870e-03
            3.99402953001282645661973447204218e-01 5.17840098623572941960446769371629e-01 8.27569483751444678887310146819800e-02 8.31683275177276352951860616258273e-03
            3.02924326461218529349395112149068e-01 6.01024825775054227783300575538306e-01 9.60508477637272983784555435704533e-02 9.05592511427729708606282343907878e-03
            2.14513913695730473740752586309100e-01 6.77253067479893777758093165175524e-01 1.08233018824375748501154248515377e-01 9.11411515004730693467749347291829e-03
            1.37791134319915020700619834315148e-01 7.43404128057339130819514139147941e-01 1.18804737622745903991017257794738e-01 8.39790320196931103224891046465928e-03
            7.58967082947862858866017177206231e-02 7.96770050912385174868290960148443e-01 1.27333240792828483733956090873107e-01 6.91057612164136438742678691937726e-03
            3.13633037996470243058411142555997e-02 8.35167147087011674244649839238264e-01 1.33469549113341301449509046506137e-01 4.75650839269026962791153678722367e-03
            6.00374098975725578597462117613759e-03 8.57032386971469550829283434723038e-01 1.36963872038773137873590712842997e-01 2.13324063577792228524332429628885e-03
            9.93996259010242688702874147566035e-01 4.71585501322903648951934130195696e-03 1.28788597652827480760651113200765e-03 1.53495158024685351071895317298655e-05
            9.68636696200352975694158885744400e-01 2.46354387551566134106906247325242e-02 6.72786504449041089515048952307552e-03 1.83470729874563623059205430543273e-04
            9.24103291705213436557642125990242e-01 5.96158083618486961396953915937047e-02 1.62808999329378673026624824160535e-02 6.76136393817317981995429487795946e-04
            8.62208865680085256855136321973987e-01 1.08233018824375804012305479773204e-01 2.95581154955389391325581982528092e-02 1.59881159799546662507452321477786e-03
            7.85486086304269637281549876206554e-01 1.68497694526671154546804132223770e-01 4.60162191690592081716459915696760e-02 2.96516670930473038442221067612081e-03
            6.97075673538781748206361044140067e-01 2.37942843638379225579271292190242e-01 6.49814828228390262143676636696910e-02 4.68818435091215846011181156427483e-03
            6.00597046998717187804572859022301e-01 3.13725462411345801250206477561733e-01 8.56774905899370109452206634159666e-02 6.58875989549553668644943371646150e-03
            5.00000000000000000000000000000000e-01 3.92743043152134763129623706845450e-01 1.07256956847865236870376293154550e-01 8.42063085019150760202055749914507e-03
            3.99402953001282645661973447204218e-01 4.71760623892924058075948323676130e-01 1.28836423105793351773229460377479e-01 9.90776284171715829196536162726261e-03
            3.02924326461218529349395112149068e-01 5.47543242665890272924400505871745e-01 1.49532430872891253237355613237014e-01 1.07882364624303419414763638428667e-02
            2.14513913695730473740752586309100e-01 6.16988391777598454979170128353871e-01 1.68497694526671071280077285337029e-01 1.08575576922022663112210594249518e-02
            1.37791134319915020700619834315148e-01 6.77253067479893666735790702659870e-01 1.84955798200191368074740694282809e-01 1.00043412890650877666853801883917e-02
            7.58967082947862858866017177206231e-02 7.25870277942421093797520370571874e-01 1.98233013762792564804726680449676e-01 8.23250284770508412857115132510444e-03
            3.13633037996470243058411142555997e-02 7.60850647549113023870859251474030e-01 2.07786048651239951823299634270370e-01 5.66638268628971708551489427918568e-03
            6.00374098975725578597462117613759e-03 7.80770231291040617271903556684265e-01 2.13226027719202071430970590881770e-01 2.54130904569368001877371909813519e-03
            9.93996259010242688702874147566035e-01 4.18506179418750862836207105033282e-03 1.81867919556980266876378138363179e-03 1.71858715358714247869835467508182e-05
            9.68636696200352975694158885744400e-01 2.18625961205404194254153793508522e-02 9.50070767910660488042573490474751e-03 2.05420446794141230238001094576816e-04
            9.24103291705213436557642125990242e-01 5.29057490539647859439931210090435e-02 2.29909592408217774983647530007147e-02 7.57026694158199398888842157617773e-04
            8.62208865680085256855136321973987e-01 9.60508477637273538896067748282803e-02 4.17402865561873892552569031977328e-02 1.79008713283271801708362858107648e-03
            7.85486086304269637281549876206554e-01 1.49532430872891392015233691381582e-01 6.49814828228389707032164324118639e-02 3.31990759867216086997920854173572e-03
            6.97075673538781748206361044140067e-01 2.11161178899235524308863887199550e-01 9.17631475619827274847750686603831e-02 5.24906029793471579775188118333062e-03
            6.00597046998717187804572859022301e-01 2.78414082476747593020149906806182e-01 1.20988870524535219175277234171517e-01 7.37701322972529865279467387040313e-03
            5.00000000000000000000000000000000e-01 3.48537836769390818592029290812206e-01 1.51462163230609181407970709187794e-01 9.42804202456428916412356500131864e-03
            3.99402953001282645661973447204218e-01 4.18661591062034321719664831107366e-01 1.81935455936683088129512952946243e-01 1.10930886418090009909453996783668e-02
            3.02924326461218529349395112149068e-01 4.85914494639546057364043463167036e-01 2.11161178899235468797712655941723e-01 1.20788986654624414668335319333892e-02
            2.14513913695730473740752586309100e-01 5.47543242665890383946702968387399e-01 2.37942843638379142312544445303502e-01 1.21565132146703915733132461696187e-02
            1.37791134319915020700619834315148e-01 6.01024825775054227783300575538306e-01 2.61184039905030807027230821404373e-01 1.12012213549587229255433129537778e-02
            7.58967082947862858866017177206231e-02 6.44169924484817135734715520811733e-01 2.79933367220396522867531530209817e-01 9.21740712737021544720406041051319e-03
            3.13633037996470243058411142555997e-02 6.75213077418241325311498712835601e-01 2.93423618782111650382660172908800e-01 6.34428643697021454883611113473307e-03
            6.00374098975725578597462117613759e-03 6.92890611744594320242640606011264e-01 3.01105647265648368460233541554771e-01 2.84534127032305250021293474560480e-03
            9.93996259010242688702874147566035e-01 3.60582910939343014369851125877631e-03 2.39791188036388115342734117518830e-03 1.83186489931768905397094227005539e-05
            9.68636696200352975694158885744400e-01 1.88367076461916811702668184125287e-02 1.25265961534553431355742958430710e-02 2.18960385744128321243787138605796e-04
            9.24103291705213436557642125990242e-01 4.55833388787718549184369010163209e-02 3.03133694160147085239209729934373e-02 8.06924819599843167958197476252735e-04
            8.62208865680085256855136321973987e-01 8.27569483751444262553675912386097e-02 5.50341859447703168894960867874033e-02 1.90807767794141307053645828517574e-03
            7.85486086304269637281549876206554e-01 1.28836423105793351773229460377479e-01 8.56774905899370109452206634159666e-02 3.53873365473008835713208952711284e-03
            6.97075673538781748206361044140067e-01 1.81935455936683060373937337317329e-01 1.20988870524535191419701618542604e-01 5.59504316910459844475189683521421e-03
            6.00597046998717187804572859022301e-01 2.39880234135137898610068418747687e-01 1.59522718866144913585358722230012e-01 7.86325649480701209526056771892399e-03
            5.00000000000000000000000000000000e-01 3.00298523499358538391135198253323e-01 1.99701476500641461608864801746677e-01 1.00494753600610265548143118508051e-02
            3.99402953001282645661973447204218e-01 3.60716812863579427972382518419181e-01 2.39880234135137981876795265634428e-01 1.18242706897548378874862606835450e-02
            3.02924326461218529349395112149068e-01 4.18661591062033988652757443560404e-01 2.78414082476747537508998675548355e-01 1.28750587024296387489430415485003e-02
            2.14513913695730473740752586309100e-01 4.71760623892923780520192167386995e-01 3.13725462411345745739055246303906e-01 1.29577890824826066212827058166113e-02
            1.37791134319915020700619834315148e-01 5.17840098623572719915841844340321e-01 3.44368767056512314894689552602358e-01 1.19395307865579058947957946656970e-02
            7.58967082947862858866017177206231e-02 5.55013708119945436969544516614405e-01 3.69089583585268221632702534407144e-01 9.82495682229828776188540473413013e-03
            3.13633037996470243058411142555997e-02 5.81760339352525579492692031635670e-01 3.86876356847827396201466854108730e-01 6.76245927408750720916108534197519e-03
            6.00374098975725578597462117613759e-03 5.96991217889323744216767408943269e-01 3.97005041120918944486106738622766e-01 3.03288709496367728246646322531888e-03
            9.93996259010242688702874147566035e-01 3.00187049487868340413854184589582e-03 3.00187049487862789298731058806879e-03 1.87014661578114748839100217603360e-05
            9.68636696200352975694158885744400e-01 1.56816518998235399084961727567133e-02 1.56816518998234843973449414988863e-02 2.23536148622115844134677908883191e-04
            9.24103291705213436557642125990242e-01 3.79483541473932817211789370048791e-02 3.79483541473932817211789370048791e-02 8.23787671856444805688901578122341e-04
            8.62208865680085256855136321973987e-01 6.88955671599573993280074546419200e-02 6.88955671599573438168562233840930e-02 1.94795206424818400335607915252467e-03
            7.85486086304269637281549876206554e-01 1.07256956847865209114800677525636e-01 1.07256956847865153603649446267809e-01 3.61268495892317232504042934237987e-03
            6.97075673538781748206361044140067e-01 1.51462163230609125896819477929967e-01 1.51462163230609125896819477929967e-01 5.71196656027839633901210802946480e-03
            6.00597046998717187804572859022301e-01 1.99701476500641406097713570488850e-01 1.99701476500641406097713570488850e-01 8.02758026984400942238639942161171e-03
            5.00000000000000000000000000000000e-01 2.49999999999999972244424384371086e-01 2.50000000000000000000000000000000e-01 1.02594860254128558002539861604419e-02
            3.99402953001282645661973447204218e-01 3.00298523499358704924588892026804e-01 3.00298523499358704924588892026804e-01 1.20713704502781044025638834682468e-02
            3.02924326461218529349395112149068e-01 3.48537836769390763080878059554379e-01 3.48537836769390763080878059554379e-01 1.31441174888501759177028560543476e-02
            2.14513913695730473740752586309100e-01 3.92743043152134763129623706845450e-01 3.92743043152134763129623706845450e-01 1.32285767414598923186197509949125e-02
            1.37791134319915020700619834315148e-01 4.31104432840042517405265698471339e-01 4.31104432840042517405265698471339e-01 1.21890392150713761099112275587686e-02
            7.58967082947862858866017177206231e-02 4.62051645852606829301123525510775e-01 4.62051645852606829301123525510775e-01 1.00302755723202102755564624203544e-02
            3.13633037996470243058411142555997e-02 4.84318348100176487847079442872200e-01 4.84318348100176487847079442872200e-01 6.90377894707361414866619853114571e-03
            6.00374098975725578597462117613759e-03 4.96998129505121344351437073783018e-01 4.96998129505121344351437073783018e-01 3.09626738238461506058651906414525e-03
            9.93996259010242688702874147566035e-01 2.39791188036393579721683444461178e-03 3.60582910939337549990901798935283e-03 1.83186489931769718548723591133864e-05
            9.68636696200352975694158885744400e-01 1.25265961534553917078316231936697e-02 1.88367076461916325980094910619300e-02 2.18960385744129269920688063422176e-04
            9.24103291705213436557642125990242e-01 3.03133694160146981155801171325948e-02 4.55833388787718618573308049235493e-02 8.06924819599846745825366678417367e-04
            8.62208865680085256855136321973987e-01 5.50341859447703515839656063235452e-02 8.27569483751443846220041677952395e-02 1.90807767794142152731340367211033e-03
            7.85486086304269637281549876206554e-01 8.56774905899370109452206634159666e-02 1.28836423105793351773229460377479e-01 3.53873365473010396964337331837669e-03
            6.97075673538781748206361044140067e-01 1.20988870524535149786338195099233e-01 1.81935455936683088129512952946243e-01 5.59504316910462273088056051051353e-03
            6.00597046998717187804572859022301e-01 1.59522718866144858074207490972185e-01 2.39880234135137954121219650005514e-01 7.86325649480704678973008725506588e-03
            5.00000000000000000000000000000000e-01 1.99701476500641295075411107973196e-01 3.00298523499358704924588892026804e-01 1.00494753600610699229012112709825e-02
            3.99402953001282645661973447204218e-01 2.39880234135137870854492803118774e-01 3.60716812863579538994684980934835e-01 1.18242706897548881944670640109507e-02
            3.02924326461218529349395112149068e-01 2.78414082476747426486696213032701e-01 4.18661591062034099675059906076058e-01 1.28750587024296959948177487831344e-02
            2.14513913695730473740752586309100e-01 3.13725462411345579205601552530425e-01 4.71760623892923947053645861160476e-01 1.29577890824826638671574130512454e-02
            1.37791134319915020700619834315148e-01 3.44368767056512203872387090086704e-01 5.17840098623572830938144306855975e-01 1.19395307865579579365000739699099e-02
            7.58967082947862858866017177206231e-02 3.69089583585268055099248840633663e-01 5.55013708119945547991846979130059e-01 9.82495682229832939524882817750040e-03
            3.13633037996470243058411142555997e-02 3.86876356847827285179164391593076e-01 5.81760339352525690514994494151324e-01 6.76245927408753669946017694769580e-03
            6.00374098975725578597462117613759e-03 3.97005041120918722441501813591458e-01 5.96991217889323966261372333974577e-01 3.03288709496369072657340204557386e-03
            9.93996259010242688702874147566035e-01 1.81867919556985991463848861826591e-03 4.18506179418745138248736381569870e-03 1.71858715358714180107199687164155e-05
            9.68636696200352975694158885744400e-01 9.50070767910666733047087006980291e-03 2.18625961205403569753702441857968e-02 2.05420446794141121817783846026373e-04
            9.24103291705213436557642125990242e-01 2.29909592408217983150464647223998e-02 5.29057490539647651273114092873584e-02 7.57026694158199073628190411966443e-04
            8.62208865680085256855136321973987e-01 4.17402865561874863997715578989300e-02 9.60508477637272567450921201270830e-02 1.79008713283271714972189059267293e-03
            7.85486086304269637281549876206554e-01 6.49814828228390956033067027419747e-02 1.49532430872891253237355613237014e-01 3.31990759867215913525573256492862e-03
            6.97075673538781748206361044140067e-01 9.17631475619828107515019155471236e-02 2.11161178899235441042137040312809e-01 5.24906029793471319566666721811998e-03
            6.00597046998717187804572859022301e-01 1.20988870524535316319791888872714e-01 2.78414082476747481997847444290528e-01 7.37701322972529605070945990519249e-03
            5.00000000000000000000000000000000e-01 1.51462163230609236919121940445621e-01 3.48537836769390763080878059554379e-01 9.42804202456428569467661304770445e-03
            3.99402953001282645661973447204218e-01 1.81935455936683282418542262348637e-01 4.18661591062034155186211137333885e-01 1.10930886418089957867749717479455e-02
            3.02924326461218529349395112149068e-01 2.11161178899235635331166349715204e-01 4.85914494639545890830589769393555e-01 1.20788986654624345279396280261608e-02
            2.14513913695730473740752586309100e-01 2.37942843638379419868300601592637e-01 5.47543242665890161902098043356091e-01 1.21565132146703863691428182391974e-02
            1.37791134319915020700619834315148e-01 2.61184039905030973560684515177854e-01 6.01024825775054116760998113022652e-01 1.12012213549587177213728850233565e-02
            7.58967082947862858866017177206231e-02 2.79933367220396800423287686498952e-01 6.44169924484816913690110595780425e-01 9.21740712737021197775710845689900e-03
            3.13633037996470243058411142555997e-02 2.93423618782111872427265097940108e-01 6.75213077418241103266893787804293e-01 6.34428643697021194675089716952243e-03
            6.00374098975725578597462117613759e-03 3.01105647265648701527140929101733e-01 6.92890611744593987175733218464302e-01 2.84534127032305119917032776299948e-03
            9.93996259010242688702874147566035e-01 1.28788597652833118611948037823822e-03 4.71585501322898011100637205572639e-03 1.53495158024685960935617340394899e-05
            9.68636696200352975694158885744400e-01 6.72786504449046900838693474611318e-03 2.46354387551565544300924415210829e-02 1.83470729874564327790617546121155e-04
            9.24103291705213436557642125990242e-01 1.62808999329378777110033382768961e-02 5.96158083618486822619075837792479e-02 6.76136393817320584080643453006587e-04
            8.62208865680085256855136321973987e-01 2.95581154955390085214972373250930e-02 1.08233018824375734623366440700920e-01 1.59881159799547291344712363070357e-03
            7.85486086304269637281549876206554e-01 4.60162191690592914383728384564165e-02 1.68497694526671071280077285337029e-01 2.96516670930474166012480452536693e-03
            6.97075673538781748206361044140067e-01 6.49814828228390539699432792986045e-02 2.37942843638379197823695676561329e-01 4.68818435091217667470830932074932e-03
            6.00597046998717187804572859022301e-01 8.56774905899370803341597024882503e-02 3.13725462411345745739055246303906e-01 6.58875989549556183993983538016437e-03
            5.00000000000000000000000000000000e-01 1.07256956847865222992588485340093e-01 3.92743043152134763129623706845450e-01 8.42063085019153882704312508167277e-03
            3.99402953001282645661973447204218e-01 1.28836423105793435039956307264220e-01 4.71760623892923947053645861160476e-01 9.90776284171719645588183311701869e-03
            3.02924326461218529349395112149068e-01 1.49532430872891364259658075752668e-01 5.47543242665890161902098043356091e-01 1.07882364624303835748397872862370e-02
            2.14513913695730473740752586309100e-01 1.68497694526671182302379747852683e-01 6.16988391777598343956867665838217e-01 1.08575576922023079445844828683221e-02
            1.37791134319915020700619834315148e-01 1.84955798200191423585891925540636e-01 6.77253067479893555713488240144216e-01 1.00043412890651259306018516781478e-02
            7.58967082947862858866017177206231e-02 1.98233013762792648071453527336416e-01 7.25870277942420982775217908056220e-01 8.23250284770511535359371890763214e-03
            3.13633037996470243058411142555997e-02 2.07786048651240007334450865528197e-01 7.60850647549113023870859251474030e-01 5.66638268628973876955834398927436e-03
            6.00374098975725578597462117613759e-03 2.13226027719202154697697437768511e-01 7.80770231291040506249601094168611e-01 2.54130904569368999343370596477598e-03
            9.93996259010242688702874147566035e-01 8.27262281141636929976401759745386e-04 5.17647870861567480083031966842100e-03 1.28847811346783169372635924343484e-05
            9.68636696200352975694158885744400e-01 4.32158520657347205301546111400057e-03 2.70417185930735513854639151531956e-02 1.54010082759304967612093428819264e-04
            9.24103291705213436557642125990242e-01 1.04578935270863444123579100164534e-02 6.54388147677002207647234399701119e-02 5.67566401679312241765340552035468e-04
            8.62208865680085256855136321973987e-01 1.89863966971688322149525163240469e-02 1.18804737622745903991017257794738e-01 1.34208386641381221399127454674272e-03
            7.85486086304269637281549876206554e-01 2.95581154955390501548606607684633e-02 1.84955798200191312563589463024982e-01 2.48903773701327412845585307366036e-03
            6.97075673538781748206361044140067e-01 4.17402865561875141553471735278436e-02 2.61184039905030751516079590146546e-01 3.93538337351413084613804826972228e-03
            6.00597046998717187804572859022301e-01 5.50341859447704695451619727464276e-02 3.44368767056512314894689552602358e-01 5.53077571272658947976186283312927e-03
            5.00000000000000000000000000000000e-01 6.88955671599574964725221093431173e-02 4.31104432840042517405265698471339e-01 7.06849563965978189977734658100417e-03
            3.99402953001282645661973447204218e-01 8.27569483751445789110334771976341e-02 5.17840098623572830938144306855975e-01 8.31683275177277393785946202342529e-03
            3.02924326461218529349395112149068e-01 9.60508477637274787896970451583911e-02 6.01024825775054005738695650506997e-01 9.05592511427730749440367929992135e-03
            2.14513913695730473740752586309100e-01 1.08233018824375970545759173546685e-01 6.77253067479893555713488240144216e-01 9.11411515004731907774182531056795e-03
            1.37791134319915020700619834315148e-01 1.18804737622746167668985606269416e-01 7.43404128057338908774909214116633e-01 8.39790320196931970586629034869475e-03
            7.58967082947862858866017177206231e-02 1.27333240792828705778561015904415e-01 7.96770050912384952823686035117134e-01 6.91057612164137306104416680341274e-03
            3.13633037996470243058411142555997e-02 1.33469549113341551249689587166358e-01 8.35167147087011452200044914206956e-01 4.75650839269027483208196471764495e-03
            6.00374098975725578597462117613759e-03 1.36963872038773387673771253503219e-01 8.57032386971469328784678509691730e-01 2.13324063577792488732853826149949e-03
            9.93996259010242688702874147566035e-01 4.55664178577066376028825711586023e-04 5.54807681118024524635767846802992e-03 9.89264452102738371110871723468350e-06
            9.68636696200352975694158885744400e-01 2.38037151964257684758075050979187e-03 2.89829322800044461572177567632025e-02 1.18245470021315180289035728922187e-04
            9.24103291705213436557642125990242e-01 5.76031032998390243871078553183906e-03 7.01363979648026575342001365243050e-02 4.35764689768807118863680472031774e-04
            8.62208865680085256855136321973987e-01 1.04578935270862889012066787586264e-02 1.27333240792828455978380475244194e-01 1.03042174089434512146423728040645e-03
            7.85486086304269637281549876206554e-01 1.62808999329378117915112511582265e-02 1.98233013762792564804726680449676e-01 1.91102706940233244350957342305719e-03
            6.97075673538781748206361044140067e-01 2.29909592408216907621909541603600e-02 2.79933367220396578378682761467644e-01 3.02149864721848807058712793605082e-03
            6.00597046998717187804572859022301e-01 3.03133694160145975016185104777833e-02 3.69089583585268221632702534407144e-01 4.24640492373423803634802098372347e-03
            5.00000000000000000000000000000000e-01 3.79483541473931360044069549530832e-02 4.62051645852606884812274756768602e-01 5.42703162208832325802410068149584e-03
            3.99402953001282645661973447204218e-01 4.55833388787717022627710150572966e-02 5.55013708119945659014149441645714e-01 6.38547671816391566174120342225251e-03
            3.02924326461218529349395112149068e-01 5.29057490539645777771760037921922e-02 6.44169924484816913690110595780425e-01 6.95293517430991495770209454008182e-03
            2.14513913695730473740752586309100e-01 5.96158083618484671561965626551682e-02 7.25870277942421093797520370571874e-01 6.99761217212008842375459138906990e-03
            1.37791134319915020700619834315148e-01 6.54388147676999848423307071243471e-02 7.96770050912385063845988497632788e-01 6.44772078242633172456832468810717e-03
            7.58967082947862858866017177206231e-02 7.01363979648023938562317880496266e-02 8.53966893740411236990439647343010e-01 5.30578457579714089092925988211391e-03
            3.13633037996470243058411142555997e-02 7.35163367751437146768722641354543e-02 8.95120359425209288772862237237860e-01 3.65193992806948702564695530270455e-03
            6.00374098975725578597462117613759e-03 7.54410441162092215705570197314955e-02 9.18555214894033467132317127834540e-01 1.63785407505009461374967383306966e-03
            9.93996259010242688702874147566035e-01 1.88297152596153815088883543005238e-04 5.81544383716115723809148008172087e-03 6.49599998128798609156814639065125e-06
            9.68636696200352975694158885744400e-01 9.83656825228955119036489129769052e-04 3.03796469744180702710067976113351e-02 7.76458276058705868264905869402526e-05
            9.24103291705213436557642125990242e-01 2.38037151964258118438944045180961e-03 7.35163367751439783548406126101327e-02 2.86144661376163149044127287368156e-04
            8.62208865680085256855136321973987e-01 4.32158520657345730786591531114027e-03 1.33469549113341273693933430877223e-01 6.76625910831095476842544922391198e-04
            7.85486086304269637281549876206554e-01 6.72786504449045599796086492005998e-03 2.07786048651239896312148403012543e-01 1.25487495084773822626167394389540e-03
            6.97075673538781748206361044140067e-01 9.50070767910664130961873041769650e-03 2.93423618782111594871508941650973e-01 1.98406554628272188342541504368910e-03
            6.00597046998717187804572859022301e-01 1.25265961534553743605968634255987e-02 3.86876356847827451712618085366557e-01 2.78839962827796654376566465316500e-03
            5.00000000000000000000000000000000e-01 1.56816518998235086834736051741856e-02 4.84318348100176487847079442872200e-01 3.56365754784786431877408041657418e-03
            3.99402953001282645661973447204218e-01 1.88367076461916568841381547372293e-02 5.81760339352525801537296956666978e-01 4.19302003155368804149238926015641e-03
            3.02924326461218529349395112149068e-01 2.18625961205403743226050039538677e-02 6.75213077418241103266893787804293e-01 4.56564133748163080567250560193315e-03
            2.14513913695730473740752586309100e-01 2.46354387551565683078802493355397e-02 7.60850647549112912848556788958376e-01 4.59497846531655915192082773046423e-03
            1.37791134319915020700619834315148e-01 2.70417185930735617938047710140381e-02 8.35167147087011452200044914206956e-01 4.23389256461848256102298293512831e-03
            7.58967082947862858866017177206231e-02 2.89829322800044496266647087168167e-02 8.95120359425209177750559774722205e-01 3.48404073671464186226498505050131e-03
            3.13633037996470243058411142555997e-02 3.03796469744180702710067976113351e-02 9.38257049225934891545364280318609e-01 2.39804449194344713319759776481988e-03
            6.00374098975725578597462117613759e-03 3.11750066470508728488919558685666e-02 9.62821252363191826262323047558311e-01 1.07549604337475171109139715497349e-03
            9.93996259010242688702874147566035e-01 3.60449058720920965850215145209745e-05 5.96769608388521884817512486165469e-03 2.83905472259312924055178403714983e-06
            9.68636696200352975694158885744400e-01 1.88297152596150670902583335042380e-04 3.11750066470508728488919558685666e-02 3.39348451645763761214819043132707e-05
            9.24103291705213436557642125990242e-01 4.55664178577059599765247677183311e-04 7.54410441162094991263131760206306e-02 1.25058552118981685741597176253492e-04
            8.62208865680085256855136321973987e-01 8.27262281141620233262945482977102e-04 1.36963872038773110118015097214084e-01 2.95717055589187307381576763631870e-04
            7.85486086304269637281549876206554e-01 1.28788597652830711683125120003979e-03 2.13226027719202043675394975252857e-01 5.48438834010235164335977930960553e-04
            6.97075673538781748206361044140067e-01 1.81867919556982456964766559082136e-03 3.01105647265648423971384772812598e-01 8.67129106424570556856112890642407e-04
            6.00597046998717187804572859022301e-01 2.39791188036389242912993502443442e-03 3.97005041120918944486106738622766e-01 1.21866058435084363698042153600909e-03
            5.00000000000000000000000000000000e-01 3.00187049487862745930644159386702e-03 4.96998129505121399862588305040845e-01 1.55748442427121281880775249817361e-03
            3.99402953001282645661973447204218e-01 3.60582910939336465788729313430849e-03 5.96991217889324077283674796490232e-01 1.83254515960601925030071246425223e-03
            3.02924326461218529349395112149068e-01 4.18506179418742969844391410561002e-03 6.92890611744594098198035680979956e-01 1.99539803543435185703702750004140e-03
            2.14513913695730473740752586309100e-01 4.71585501322894801862206648479514e-03 7.80770231291040617271903556684265e-01 2.00821972748592078583662612345506e-03
            1.37791134319915020700619834315148e-01 5.17647870861563490219037220185783e-03 8.57032386971469439806980972207384e-01 1.85040836132341519892907655275849e-03
            7.58967082947862858866017177206231e-02 5.54807681118019754146208910583482e-03 9.18555214894033467132317127834540e-01 1.52268816745214146876619132342512e-03
            3.13633037996470243058411142555997e-02 5.81544383716110519638720077750804e-03 9.62821252363191826262323047558311e-01 1.04805719819146326600434182552135e-03
            6.00374098975725578597462117613759e-03 5.96769608388516420438563159223122e-03 9.88028562926357545315170227695489e-01 4.70041891913291847403255641069109e-04
            ];
        
    case  28
        
        % ALG. DEG.:   28
        % PTS CARD.:  225
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.3e-15
        
        xyw_bar=[
            9.93996259010242688702874147566035e-01 5.96769608388527435932635611948172e-03 3.60449058720369377994963144828944e-05 2.83905472259312966406825766430000e-06
            9.68636696200352975694158885744400e-01 3.11750066470509283600431871263936e-02 1.88297152596095945797927129206073e-04 3.39348451645763828977454823476734e-05
            9.24103291705213436557642125990242e-01 7.54410441162094991263131760206306e-02 4.55664178577064316044697989127599e-04 1.25058552118981712846651488391103e-04
            8.62208865680085256855136321973987e-01 1.36963872038773165629166328471911e-01 8.27262281141577515697349554102402e-04 2.95717055589187361591685387907091e-04
            7.85486086304269637281549876206554e-01 2.13226027719202099186546206510684e-01 1.28788597652826353190391728276154e-03 5.48438834010235272756195179510996e-04
            6.97075673538781748206361044140067e-01 3.01105647265648423971384772812598e-01 1.81867919556982782225418304733466e-03 8.67129106424570773696547387743294e-04
            6.00597046998717187804572859022301e-01 3.97005041120918888974955507364939e-01 2.39791188036392322047163361276034e-03 1.21866058435084385382085603310998e-03
            5.00000000000000000000000000000000e-01 4.96998129505121288840285842525191e-01 3.00187049487871115971415747480933e-03 1.55748442427121325248862149237539e-03
            3.99402953001282645661973447204218e-01 5.96991217889323966261372333974577e-01 3.60582910939344358780545007903129e-03 1.83254515960601968398158145845400e-03
            3.02924326461218529349395112149068e-01 6.92890611744594098198035680979956e-01 4.18506179418742796372043812880293e-03 1.99539803543435229071789649424318e-03
            2.14513913695730473740752586309100e-01 7.80770231291040506249601094168611e-01 4.71585501322902000964631952228956e-03 2.00821972748592121951749511765684e-03
            1.37791134319915020700619834315148e-01 8.57032386971469328784678509691730e-01 5.17647870861570602585288725094870e-03 1.85040836132341563260994554696026e-03
            7.58967082947862858866017177206231e-02 9.18555214894033356110014665318886e-01 5.54807681118030249223238570266403e-03 1.52268816745214168560662582052601e-03
            3.13633037996470243058411142555997e-02 9.62821252363191826262323047558311e-01 5.81544383716114943183583818608895e-03 1.04805719819146348284477632262224e-03
            6.00374098975725578597462117613759e-03 9.88028562926357434292867765179835e-01 5.96769608388525441000638238620013e-03 4.70041891913291955823472889619552e-04
            9.93996259010242688702874147566035e-01 5.81544383716121274924271133954790e-03 1.88297152596098547883141094416715e-04 6.49599998128798524453519913635091e-06
            9.68636696200352975694158885744400e-01 3.03796469744181257821580288691621e-02 9.83656825228898523683085386437597e-04 7.76458276058705732739634308714471e-05
            9.24103291705213436557642125990242e-01 7.35163367751439783548406126101327e-02 2.38037151964258508751726139962557e-03 2.86144661376163094834018663092934e-04
            8.62208865680085256855136321973987e-01 1.33469549113341329205084662135050e-01 4.32158520657341393977901589096291e-03 6.76625910831095368422327673840755e-04
            7.85486086304269637281549876206554e-01 2.07786048651239951823299634270370e-01 6.72786504449041089515048952307552e-03 1.25487495084773800942123944679452e-03
            6.97075673538781748206361044140067e-01 2.93423618782111594871508941650973e-01 9.50070767910665692213001420896035e-03 1.98406554628272188342541504368910e-03
            6.00597046998717187804572859022301e-01 3.86876356847827451712618085366557e-01 1.25265961534553604828090556111420e-02 2.78839962827796611008479565896323e-03
            5.00000000000000000000000000000000e-01 4.84318348100176432335928211614373e-01 1.56816518998235676640717883856269e-02 3.56365754784786345141234242817063e-03
            3.99402953001282645661973447204218e-01 5.81760339352525801537296956666978e-01 1.88367076461916083118808273866307e-02 4.19302003155368717413065127175287e-03
            3.02924326461218529349395112149068e-01 6.75213077418241103266893787804293e-01 2.18625961205404228948623313044664e-02 4.56564133748162907094902962512606e-03
            2.14513913695730473740752586309100e-01 7.60850647549112912848556788958376e-01 2.46354387551566134106906247325242e-02 4.59497846531655828455908974206068e-03
            1.37791134319915020700619834315148e-01 8.35167147087011452200044914206956e-01 2.70417185930735826104864827357233e-02 4.23389256461848169366124494672476e-03
            7.58967082947862858866017177206231e-02 8.95120359425209177750559774722205e-01 2.89829322800044808516872762993444e-02 3.48404073671464099490324706209776e-03
            3.13633037996470243058411142555997e-02 9.38257049225934891545364280318609e-01 3.03796469744180841487946054257918e-02 2.39804449194344669951672877061810e-03
            6.00374098975725578597462117613759e-03 9.62821252363191826262323047558311e-01 3.11750066470508624405511000077240e-02 1.07549604337475149425096265787261e-03
            9.93996259010242688702874147566035e-01 5.54807681118029902278543374904984e-03 4.55664178577012274340418684914766e-04 9.89264452102744639154681405290859e-06
            9.68636696200352975694158885744400e-01 2.89829322800044947294750841138011e-02 2.38037151964252957636603014179855e-03 1.18245470021315912125502156637680e-04
            9.24103291705213436557642125990242e-01 7.01363979648026436564123287098482e-02 5.76031032998391978594554529991001e-03 4.35764689768809829369111685792859e-04
            8.62208865680085256855136321973987e-01 1.27333240792828483733956090873107e-01 1.04578935270862594109075871529058e-02 1.03042174089435162667727219343305e-03
            7.85486086304269637281549876206554e-01 1.98233013762792537049151064820762e-01 1.62808999329378256692990589726833e-02 1.91102706940234436973347076360596e-03
            6.97075673538781748206361044140067e-01 2.79933367220396467356380298951990e-01 2.29909592408217844372586569079431e-02 3.02149864721850671886449468672708e-03
            6.00597046998717187804572859022301e-01 3.69089583585268110610400071891490e-01 3.03133694160147015850270690862089e-02 4.24640492373426405720016063582989e-03
            5.00000000000000000000000000000000e-01 4.62051645852606662767669831737294e-01 3.79483541473933372323301682627061e-02 5.42703162208835708513188222923418e-03
            3.99402953001282645661973447204218e-01 5.55013708119945547991846979130059e-01 4.55833388787718618573308049235493e-02 6.38547671816395469301941290041214e-03
            3.02924326461218529349395112149068e-01 6.44169924484816802667808133264771e-01 5.29057490539647234939479858439881e-02 6.95293517430995832578899396025918e-03
            2.14513913695730473740752586309100e-01 7.25870277942420871752915445540566e-01 5.96158083618486545063319681503344e-02 6.99761217212013179184149080924726e-03
            1.37791134319915020700619834315148e-01 7.96770050912384841801383572601480e-01 6.54388147677001930091478243411984e-02 6.44772078242637249057001014307389e-03
            7.58967082947862858866017177206231e-02 8.53966893740411014945834722311702e-01 7.01363979648026436564123287098482e-02 5.30578457579717385067530344144870e-03
            3.13633037996470243058411142555997e-02 8.95120359425208955705954849690897e-01 7.35163367751440199882040360535029e-02 3.65193992806951001073301199539856e-03
            6.00374098975725578597462117613759e-03 9.18555214894033134065409740287578e-01 7.54410441162095546374644072784577e-02 1.63785407505010480525009519681134e-03
            9.93996259010242688702874147566035e-01 5.17647870861573117934328891465157e-03 8.27262281141580117782563519313044e-04 1.28847811346783016906705418569423e-05
            9.68636696200352975694158885744400e-01 2.70417185930736173049560022718651e-02 4.32158520657340700088511198373453e-03 1.54010082759304777876713243855988e-04
            9.24103291705213436557642125990242e-01 6.54388147677002346425112477845687e-02 1.04578935270863287998466262251895e-02 5.67566401679311591244037060732808e-04
            8.62208865680085256855136321973987e-01 1.18804737622746001135531912495935e-01 1.89863966971687420093317655300780e-02 1.34208386641381069610823306703651e-03
            7.85486086304269637281549876206554e-01 1.84955798200191423585891925540636e-01 2.95581154955389391325581982528092e-02 2.48903773701327109268977011424795e-03
            6.97075673538781748206361044140067e-01 2.61184039905030807027230821404373e-01 4.17402865561874447664081344555598e-02 3.93538337351412650932935832770454e-03
            6.00597046998717187804572859022301e-01 3.44368767056512481428143246375839e-01 5.50341859447703307672838946018601e-02 5.53077571272658340822969691430444e-03
            5.00000000000000000000000000000000e-01 4.31104432840042572916416929729166e-01 6.88955671599574270835830702708336e-02 7.06849563965977322615996669696870e-03
            3.99402953001282645661973447204218e-01 5.17840098623572941960446769371629e-01 8.27569483751444678887310146819800e-02 8.31683275177276352951860616258273e-03
            3.02924326461218529349395112149068e-01 6.01024825775054227783300575538306e-01 9.60508477637272983784555435704533e-02 9.05592511427729708606282343907878e-03
            2.14513913695730473740752586309100e-01 6.77253067479893777758093165175524e-01 1.08233018824375748501154248515377e-01 9.11411515004730693467749347291829e-03
            1.37791134319915020700619834315148e-01 7.43404128057339130819514139147941e-01 1.18804737622745903991017257794738e-01 8.39790320196931103224891046465928e-03
            7.58967082947862858866017177206231e-02 7.96770050912385174868290960148443e-01 1.27333240792828483733956090873107e-01 6.91057612164136438742678691937726e-03
            3.13633037996470243058411142555997e-02 8.35167147087011674244649839238264e-01 1.33469549113341301449509046506137e-01 4.75650839269026962791153678722367e-03
            6.00374098975725578597462117613759e-03 8.57032386971469550829283434723038e-01 1.36963872038773137873590712842997e-01 2.13324063577792228524332429628885e-03
            9.93996259010242688702874147566035e-01 4.71585501322903648951934130195696e-03 1.28788597652827480760651113200765e-03 1.53495158024685351071895317298655e-05
            9.68636696200352975694158885744400e-01 2.46354387551566134106906247325242e-02 6.72786504449041089515048952307552e-03 1.83470729874563623059205430543273e-04
            9.24103291705213436557642125990242e-01 5.96158083618486961396953915937047e-02 1.62808999329378673026624824160535e-02 6.76136393817317981995429487795946e-04
            8.62208865680085256855136321973987e-01 1.08233018824375804012305479773204e-01 2.95581154955389391325581982528092e-02 1.59881159799546662507452321477786e-03
            7.85486086304269637281549876206554e-01 1.68497694526671154546804132223770e-01 4.60162191690592081716459915696760e-02 2.96516670930473038442221067612081e-03
            6.97075673538781748206361044140067e-01 2.37942843638379225579271292190242e-01 6.49814828228390262143676636696910e-02 4.68818435091215846011181156427483e-03
            6.00597046998717187804572859022301e-01 3.13725462411345801250206477561733e-01 8.56774905899370109452206634159666e-02 6.58875989549553668644943371646150e-03
            5.00000000000000000000000000000000e-01 3.92743043152134763129623706845450e-01 1.07256956847865236870376293154550e-01 8.42063085019150760202055749914507e-03
            3.99402953001282645661973447204218e-01 4.71760623892924058075948323676130e-01 1.28836423105793351773229460377479e-01 9.90776284171715829196536162726261e-03
            3.02924326461218529349395112149068e-01 5.47543242665890272924400505871745e-01 1.49532430872891253237355613237014e-01 1.07882364624303419414763638428667e-02
            2.14513913695730473740752586309100e-01 6.16988391777598454979170128353871e-01 1.68497694526671071280077285337029e-01 1.08575576922022663112210594249518e-02
            1.37791134319915020700619834315148e-01 6.77253067479893666735790702659870e-01 1.84955798200191368074740694282809e-01 1.00043412890650877666853801883917e-02
            7.58967082947862858866017177206231e-02 7.25870277942421093797520370571874e-01 1.98233013762792564804726680449676e-01 8.23250284770508412857115132510444e-03
            3.13633037996470243058411142555997e-02 7.60850647549113023870859251474030e-01 2.07786048651239951823299634270370e-01 5.66638268628971708551489427918568e-03
            6.00374098975725578597462117613759e-03 7.80770231291040617271903556684265e-01 2.13226027719202071430970590881770e-01 2.54130904569368001877371909813519e-03
            9.93996259010242688702874147566035e-01 4.18506179418750862836207105033282e-03 1.81867919556980266876378138363179e-03 1.71858715358714247869835467508182e-05
            9.68636696200352975694158885744400e-01 2.18625961205404194254153793508522e-02 9.50070767910660488042573490474751e-03 2.05420446794141230238001094576816e-04
            9.24103291705213436557642125990242e-01 5.29057490539647859439931210090435e-02 2.29909592408217774983647530007147e-02 7.57026694158199398888842157617773e-04
            8.62208865680085256855136321973987e-01 9.60508477637273538896067748282803e-02 4.17402865561873892552569031977328e-02 1.79008713283271801708362858107648e-03
            7.85486086304269637281549876206554e-01 1.49532430872891392015233691381582e-01 6.49814828228389707032164324118639e-02 3.31990759867216086997920854173572e-03
            6.97075673538781748206361044140067e-01 2.11161178899235524308863887199550e-01 9.17631475619827274847750686603831e-02 5.24906029793471579775188118333062e-03
            6.00597046998717187804572859022301e-01 2.78414082476747593020149906806182e-01 1.20988870524535219175277234171517e-01 7.37701322972529865279467387040313e-03
            5.00000000000000000000000000000000e-01 3.48537836769390818592029290812206e-01 1.51462163230609181407970709187794e-01 9.42804202456428916412356500131864e-03
            3.99402953001282645661973447204218e-01 4.18661591062034321719664831107366e-01 1.81935455936683088129512952946243e-01 1.10930886418090009909453996783668e-02
            3.02924326461218529349395112149068e-01 4.85914494639546057364043463167036e-01 2.11161178899235468797712655941723e-01 1.20788986654624414668335319333892e-02
            2.14513913695730473740752586309100e-01 5.47543242665890383946702968387399e-01 2.37942843638379142312544445303502e-01 1.21565132146703915733132461696187e-02
            1.37791134319915020700619834315148e-01 6.01024825775054227783300575538306e-01 2.61184039905030807027230821404373e-01 1.12012213549587229255433129537778e-02
            7.58967082947862858866017177206231e-02 6.44169924484817135734715520811733e-01 2.79933367220396522867531530209817e-01 9.21740712737021544720406041051319e-03
            3.13633037996470243058411142555997e-02 6.75213077418241325311498712835601e-01 2.93423618782111650382660172908800e-01 6.34428643697021454883611113473307e-03
            6.00374098975725578597462117613759e-03 6.92890611744594320242640606011264e-01 3.01105647265648368460233541554771e-01 2.84534127032305250021293474560480e-03
            9.93996259010242688702874147566035e-01 3.60582910939343014369851125877631e-03 2.39791188036388115342734117518830e-03 1.83186489931768905397094227005539e-05
            9.68636696200352975694158885744400e-01 1.88367076461916811702668184125287e-02 1.25265961534553431355742958430710e-02 2.18960385744128321243787138605796e-04
            9.24103291705213436557642125990242e-01 4.55833388787718549184369010163209e-02 3.03133694160147085239209729934373e-02 8.06924819599843167958197476252735e-04
            8.62208865680085256855136321973987e-01 8.27569483751444262553675912386097e-02 5.50341859447703168894960867874033e-02 1.90807767794141307053645828517574e-03
            7.85486086304269637281549876206554e-01 1.28836423105793351773229460377479e-01 8.56774905899370109452206634159666e-02 3.53873365473008835713208952711284e-03
            6.97075673538781748206361044140067e-01 1.81935455936683060373937337317329e-01 1.20988870524535191419701618542604e-01 5.59504316910459844475189683521421e-03
            6.00597046998717187804572859022301e-01 2.39880234135137898610068418747687e-01 1.59522718866144913585358722230012e-01 7.86325649480701209526056771892399e-03
            5.00000000000000000000000000000000e-01 3.00298523499358538391135198253323e-01 1.99701476500641461608864801746677e-01 1.00494753600610265548143118508051e-02
            3.99402953001282645661973447204218e-01 3.60716812863579427972382518419181e-01 2.39880234135137981876795265634428e-01 1.18242706897548378874862606835450e-02
            3.02924326461218529349395112149068e-01 4.18661591062033988652757443560404e-01 2.78414082476747537508998675548355e-01 1.28750587024296387489430415485003e-02
            2.14513913695730473740752586309100e-01 4.71760623892923780520192167386995e-01 3.13725462411345745739055246303906e-01 1.29577890824826066212827058166113e-02
            1.37791134319915020700619834315148e-01 5.17840098623572719915841844340321e-01 3.44368767056512314894689552602358e-01 1.19395307865579058947957946656970e-02
            7.58967082947862858866017177206231e-02 5.55013708119945436969544516614405e-01 3.69089583585268221632702534407144e-01 9.82495682229828776188540473413013e-03
            3.13633037996470243058411142555997e-02 5.81760339352525579492692031635670e-01 3.86876356847827396201466854108730e-01 6.76245927408750720916108534197519e-03
            6.00374098975725578597462117613759e-03 5.96991217889323744216767408943269e-01 3.97005041120918944486106738622766e-01 3.03288709496367728246646322531888e-03
            9.93996259010242688702874147566035e-01 3.00187049487868340413854184589582e-03 3.00187049487862789298731058806879e-03 1.87014661578114748839100217603360e-05
            9.68636696200352975694158885744400e-01 1.56816518998235399084961727567133e-02 1.56816518998234843973449414988863e-02 2.23536148622115844134677908883191e-04
            9.24103291705213436557642125990242e-01 3.79483541473932817211789370048791e-02 3.79483541473932817211789370048791e-02 8.23787671856444805688901578122341e-04
            8.62208865680085256855136321973987e-01 6.88955671599573993280074546419200e-02 6.88955671599573438168562233840930e-02 1.94795206424818400335607915252467e-03
            7.85486086304269637281549876206554e-01 1.07256956847865209114800677525636e-01 1.07256956847865153603649446267809e-01 3.61268495892317232504042934237987e-03
            6.97075673538781748206361044140067e-01 1.51462163230609125896819477929967e-01 1.51462163230609125896819477929967e-01 5.71196656027839633901210802946480e-03
            6.00597046998717187804572859022301e-01 1.99701476500641406097713570488850e-01 1.99701476500641406097713570488850e-01 8.02758026984400942238639942161171e-03
            5.00000000000000000000000000000000e-01 2.49999999999999972244424384371086e-01 2.50000000000000000000000000000000e-01 1.02594860254128558002539861604419e-02
            3.99402953001282645661973447204218e-01 3.00298523499358704924588892026804e-01 3.00298523499358704924588892026804e-01 1.20713704502781044025638834682468e-02
            3.02924326461218529349395112149068e-01 3.48537836769390763080878059554379e-01 3.48537836769390763080878059554379e-01 1.31441174888501759177028560543476e-02
            2.14513913695730473740752586309100e-01 3.92743043152134763129623706845450e-01 3.92743043152134763129623706845450e-01 1.32285767414598923186197509949125e-02
            1.37791134319915020700619834315148e-01 4.31104432840042517405265698471339e-01 4.31104432840042517405265698471339e-01 1.21890392150713761099112275587686e-02
            7.58967082947862858866017177206231e-02 4.62051645852606829301123525510775e-01 4.62051645852606829301123525510775e-01 1.00302755723202102755564624203544e-02
            3.13633037996470243058411142555997e-02 4.84318348100176487847079442872200e-01 4.84318348100176487847079442872200e-01 6.90377894707361414866619853114571e-03
            6.00374098975725578597462117613759e-03 4.96998129505121344351437073783018e-01 4.96998129505121344351437073783018e-01 3.09626738238461506058651906414525e-03
            9.93996259010242688702874147566035e-01 2.39791188036393579721683444461178e-03 3.60582910939337549990901798935283e-03 1.83186489931769718548723591133864e-05
            9.68636696200352975694158885744400e-01 1.25265961534553917078316231936697e-02 1.88367076461916325980094910619300e-02 2.18960385744129269920688063422176e-04
            9.24103291705213436557642125990242e-01 3.03133694160146981155801171325948e-02 4.55833388787718618573308049235493e-02 8.06924819599846745825366678417367e-04
            8.62208865680085256855136321973987e-01 5.50341859447703515839656063235452e-02 8.27569483751443846220041677952395e-02 1.90807767794142152731340367211033e-03
            7.85486086304269637281549876206554e-01 8.56774905899370109452206634159666e-02 1.28836423105793351773229460377479e-01 3.53873365473010396964337331837669e-03
            6.97075673538781748206361044140067e-01 1.20988870524535149786338195099233e-01 1.81935455936683088129512952946243e-01 5.59504316910462273088056051051353e-03
            6.00597046998717187804572859022301e-01 1.59522718866144858074207490972185e-01 2.39880234135137954121219650005514e-01 7.86325649480704678973008725506588e-03
            5.00000000000000000000000000000000e-01 1.99701476500641295075411107973196e-01 3.00298523499358704924588892026804e-01 1.00494753600610699229012112709825e-02
            3.99402953001282645661973447204218e-01 2.39880234135137870854492803118774e-01 3.60716812863579538994684980934835e-01 1.18242706897548881944670640109507e-02
            3.02924326461218529349395112149068e-01 2.78414082476747426486696213032701e-01 4.18661591062034099675059906076058e-01 1.28750587024296959948177487831344e-02
            2.14513913695730473740752586309100e-01 3.13725462411345579205601552530425e-01 4.71760623892923947053645861160476e-01 1.29577890824826638671574130512454e-02
            1.37791134319915020700619834315148e-01 3.44368767056512203872387090086704e-01 5.17840098623572830938144306855975e-01 1.19395307865579579365000739699099e-02
            7.58967082947862858866017177206231e-02 3.69089583585268055099248840633663e-01 5.55013708119945547991846979130059e-01 9.82495682229832939524882817750040e-03
            3.13633037996470243058411142555997e-02 3.86876356847827285179164391593076e-01 5.81760339352525690514994494151324e-01 6.76245927408753669946017694769580e-03
            6.00374098975725578597462117613759e-03 3.97005041120918722441501813591458e-01 5.96991217889323966261372333974577e-01 3.03288709496369072657340204557386e-03
            9.93996259010242688702874147566035e-01 1.81867919556985991463848861826591e-03 4.18506179418745138248736381569870e-03 1.71858715358714180107199687164155e-05
            9.68636696200352975694158885744400e-01 9.50070767910666733047087006980291e-03 2.18625961205403569753702441857968e-02 2.05420446794141121817783846026373e-04
            9.24103291705213436557642125990242e-01 2.29909592408217983150464647223998e-02 5.29057490539647651273114092873584e-02 7.57026694158199073628190411966443e-04
            8.62208865680085256855136321973987e-01 4.17402865561874863997715578989300e-02 9.60508477637272567450921201270830e-02 1.79008713283271714972189059267293e-03
            7.85486086304269637281549876206554e-01 6.49814828228390956033067027419747e-02 1.49532430872891253237355613237014e-01 3.31990759867215913525573256492862e-03
            6.97075673538781748206361044140067e-01 9.17631475619828107515019155471236e-02 2.11161178899235441042137040312809e-01 5.24906029793471319566666721811998e-03
            6.00597046998717187804572859022301e-01 1.20988870524535316319791888872714e-01 2.78414082476747481997847444290528e-01 7.37701322972529605070945990519249e-03
            5.00000000000000000000000000000000e-01 1.51462163230609236919121940445621e-01 3.48537836769390763080878059554379e-01 9.42804202456428569467661304770445e-03
            3.99402953001282645661973447204218e-01 1.81935455936683282418542262348637e-01 4.18661591062034155186211137333885e-01 1.10930886418089957867749717479455e-02
            3.02924326461218529349395112149068e-01 2.11161178899235635331166349715204e-01 4.85914494639545890830589769393555e-01 1.20788986654624345279396280261608e-02
            2.14513913695730473740752586309100e-01 2.37942843638379419868300601592637e-01 5.47543242665890161902098043356091e-01 1.21565132146703863691428182391974e-02
            1.37791134319915020700619834315148e-01 2.61184039905030973560684515177854e-01 6.01024825775054116760998113022652e-01 1.12012213549587177213728850233565e-02
            7.58967082947862858866017177206231e-02 2.79933367220396800423287686498952e-01 6.44169924484816913690110595780425e-01 9.21740712737021197775710845689900e-03
            3.13633037996470243058411142555997e-02 2.93423618782111872427265097940108e-01 6.75213077418241103266893787804293e-01 6.34428643697021194675089716952243e-03
            6.00374098975725578597462117613759e-03 3.01105647265648701527140929101733e-01 6.92890611744593987175733218464302e-01 2.84534127032305119917032776299948e-03
            9.93996259010242688702874147566035e-01 1.28788597652833118611948037823822e-03 4.71585501322898011100637205572639e-03 1.53495158024685960935617340394899e-05
            9.68636696200352975694158885744400e-01 6.72786504449046900838693474611318e-03 2.46354387551565544300924415210829e-02 1.83470729874564327790617546121155e-04
            9.24103291705213436557642125990242e-01 1.62808999329378777110033382768961e-02 5.96158083618486822619075837792479e-02 6.76136393817320584080643453006587e-04
            8.62208865680085256855136321973987e-01 2.95581154955390085214972373250930e-02 1.08233018824375734623366440700920e-01 1.59881159799547291344712363070357e-03
            7.85486086304269637281549876206554e-01 4.60162191690592914383728384564165e-02 1.68497694526671071280077285337029e-01 2.96516670930474166012480452536693e-03
            6.97075673538781748206361044140067e-01 6.49814828228390539699432792986045e-02 2.37942843638379197823695676561329e-01 4.68818435091217667470830932074932e-03
            6.00597046998717187804572859022301e-01 8.56774905899370803341597024882503e-02 3.13725462411345745739055246303906e-01 6.58875989549556183993983538016437e-03
            5.00000000000000000000000000000000e-01 1.07256956847865222992588485340093e-01 3.92743043152134763129623706845450e-01 8.42063085019153882704312508167277e-03
            3.99402953001282645661973447204218e-01 1.28836423105793435039956307264220e-01 4.71760623892923947053645861160476e-01 9.90776284171719645588183311701869e-03
            3.02924326461218529349395112149068e-01 1.49532430872891364259658075752668e-01 5.47543242665890161902098043356091e-01 1.07882364624303835748397872862370e-02
            2.14513913695730473740752586309100e-01 1.68497694526671182302379747852683e-01 6.16988391777598343956867665838217e-01 1.08575576922023079445844828683221e-02
            1.37791134319915020700619834315148e-01 1.84955798200191423585891925540636e-01 6.77253067479893555713488240144216e-01 1.00043412890651259306018516781478e-02
            7.58967082947862858866017177206231e-02 1.98233013762792648071453527336416e-01 7.25870277942420982775217908056220e-01 8.23250284770511535359371890763214e-03
            3.13633037996470243058411142555997e-02 2.07786048651240007334450865528197e-01 7.60850647549113023870859251474030e-01 5.66638268628973876955834398927436e-03
            6.00374098975725578597462117613759e-03 2.13226027719202154697697437768511e-01 7.80770231291040506249601094168611e-01 2.54130904569368999343370596477598e-03
            9.93996259010242688702874147566035e-01 8.27262281141636929976401759745386e-04 5.17647870861567480083031966842100e-03 1.28847811346783169372635924343484e-05
            9.68636696200352975694158885744400e-01 4.32158520657347205301546111400057e-03 2.70417185930735513854639151531956e-02 1.54010082759304967612093428819264e-04
            9.24103291705213436557642125990242e-01 1.04578935270863444123579100164534e-02 6.54388147677002207647234399701119e-02 5.67566401679312241765340552035468e-04
            8.62208865680085256855136321973987e-01 1.89863966971688322149525163240469e-02 1.18804737622745903991017257794738e-01 1.34208386641381221399127454674272e-03
            7.85486086304269637281549876206554e-01 2.95581154955390501548606607684633e-02 1.84955798200191312563589463024982e-01 2.48903773701327412845585307366036e-03
            6.97075673538781748206361044140067e-01 4.17402865561875141553471735278436e-02 2.61184039905030751516079590146546e-01 3.93538337351413084613804826972228e-03
            6.00597046998717187804572859022301e-01 5.50341859447704695451619727464276e-02 3.44368767056512314894689552602358e-01 5.53077571272658947976186283312927e-03
            5.00000000000000000000000000000000e-01 6.88955671599574964725221093431173e-02 4.31104432840042517405265698471339e-01 7.06849563965978189977734658100417e-03
            3.99402953001282645661973447204218e-01 8.27569483751445789110334771976341e-02 5.17840098623572830938144306855975e-01 8.31683275177277393785946202342529e-03
            3.02924326461218529349395112149068e-01 9.60508477637274787896970451583911e-02 6.01024825775054005738695650506997e-01 9.05592511427730749440367929992135e-03
            2.14513913695730473740752586309100e-01 1.08233018824375970545759173546685e-01 6.77253067479893555713488240144216e-01 9.11411515004731907774182531056795e-03
            1.37791134319915020700619834315148e-01 1.18804737622746167668985606269416e-01 7.43404128057338908774909214116633e-01 8.39790320196931970586629034869475e-03
            7.58967082947862858866017177206231e-02 1.27333240792828705778561015904415e-01 7.96770050912384952823686035117134e-01 6.91057612164137306104416680341274e-03
            3.13633037996470243058411142555997e-02 1.33469549113341551249689587166358e-01 8.35167147087011452200044914206956e-01 4.75650839269027483208196471764495e-03
            6.00374098975725578597462117613759e-03 1.36963872038773387673771253503219e-01 8.57032386971469328784678509691730e-01 2.13324063577792488732853826149949e-03
            9.93996259010242688702874147566035e-01 4.55664178577066376028825711586023e-04 5.54807681118024524635767846802992e-03 9.89264452102738371110871723468350e-06
            9.68636696200352975694158885744400e-01 2.38037151964257684758075050979187e-03 2.89829322800044461572177567632025e-02 1.18245470021315180289035728922187e-04
            9.24103291705213436557642125990242e-01 5.76031032998390243871078553183906e-03 7.01363979648026575342001365243050e-02 4.35764689768807118863680472031774e-04
            8.62208865680085256855136321973987e-01 1.04578935270862889012066787586264e-02 1.27333240792828455978380475244194e-01 1.03042174089434512146423728040645e-03
            7.85486086304269637281549876206554e-01 1.62808999329378117915112511582265e-02 1.98233013762792564804726680449676e-01 1.91102706940233244350957342305719e-03
            6.97075673538781748206361044140067e-01 2.29909592408216907621909541603600e-02 2.79933367220396578378682761467644e-01 3.02149864721848807058712793605082e-03
            6.00597046998717187804572859022301e-01 3.03133694160145975016185104777833e-02 3.69089583585268221632702534407144e-01 4.24640492373423803634802098372347e-03
            5.00000000000000000000000000000000e-01 3.79483541473931360044069549530832e-02 4.62051645852606884812274756768602e-01 5.42703162208832325802410068149584e-03
            3.99402953001282645661973447204218e-01 4.55833388787717022627710150572966e-02 5.55013708119945659014149441645714e-01 6.38547671816391566174120342225251e-03
            3.02924326461218529349395112149068e-01 5.29057490539645777771760037921922e-02 6.44169924484816913690110595780425e-01 6.95293517430991495770209454008182e-03
            2.14513913695730473740752586309100e-01 5.96158083618484671561965626551682e-02 7.25870277942421093797520370571874e-01 6.99761217212008842375459138906990e-03
            1.37791134319915020700619834315148e-01 6.54388147676999848423307071243471e-02 7.96770050912385063845988497632788e-01 6.44772078242633172456832468810717e-03
            7.58967082947862858866017177206231e-02 7.01363979648023938562317880496266e-02 8.53966893740411236990439647343010e-01 5.30578457579714089092925988211391e-03
            3.13633037996470243058411142555997e-02 7.35163367751437146768722641354543e-02 8.95120359425209288772862237237860e-01 3.65193992806948702564695530270455e-03
            6.00374098975725578597462117613759e-03 7.54410441162092215705570197314955e-02 9.18555214894033467132317127834540e-01 1.63785407505009461374967383306966e-03
            9.93996259010242688702874147566035e-01 1.88297152596153815088883543005238e-04 5.81544383716115723809148008172087e-03 6.49599998128798609156814639065125e-06
            9.68636696200352975694158885744400e-01 9.83656825228955119036489129769052e-04 3.03796469744180702710067976113351e-02 7.76458276058705868264905869402526e-05
            9.24103291705213436557642125990242e-01 2.38037151964258118438944045180961e-03 7.35163367751439783548406126101327e-02 2.86144661376163149044127287368156e-04
            8.62208865680085256855136321973987e-01 4.32158520657345730786591531114027e-03 1.33469549113341273693933430877223e-01 6.76625910831095476842544922391198e-04
            7.85486086304269637281549876206554e-01 6.72786504449045599796086492005998e-03 2.07786048651239896312148403012543e-01 1.25487495084773822626167394389540e-03
            6.97075673538781748206361044140067e-01 9.50070767910664130961873041769650e-03 2.93423618782111594871508941650973e-01 1.98406554628272188342541504368910e-03
            6.00597046998717187804572859022301e-01 1.25265961534553743605968634255987e-02 3.86876356847827451712618085366557e-01 2.78839962827796654376566465316500e-03
            5.00000000000000000000000000000000e-01 1.56816518998235086834736051741856e-02 4.84318348100176487847079442872200e-01 3.56365754784786431877408041657418e-03
            3.99402953001282645661973447204218e-01 1.88367076461916568841381547372293e-02 5.81760339352525801537296956666978e-01 4.19302003155368804149238926015641e-03
            3.02924326461218529349395112149068e-01 2.18625961205403743226050039538677e-02 6.75213077418241103266893787804293e-01 4.56564133748163080567250560193315e-03
            2.14513913695730473740752586309100e-01 2.46354387551565683078802493355397e-02 7.60850647549112912848556788958376e-01 4.59497846531655915192082773046423e-03
            1.37791134319915020700619834315148e-01 2.70417185930735617938047710140381e-02 8.35167147087011452200044914206956e-01 4.23389256461848256102298293512831e-03
            7.58967082947862858866017177206231e-02 2.89829322800044496266647087168167e-02 8.95120359425209177750559774722205e-01 3.48404073671464186226498505050131e-03
            3.13633037996470243058411142555997e-02 3.03796469744180702710067976113351e-02 9.38257049225934891545364280318609e-01 2.39804449194344713319759776481988e-03
            6.00374098975725578597462117613759e-03 3.11750066470508728488919558685666e-02 9.62821252363191826262323047558311e-01 1.07549604337475171109139715497349e-03
            9.93996259010242688702874147566035e-01 3.60449058720920965850215145209745e-05 5.96769608388521884817512486165469e-03 2.83905472259312924055178403714983e-06
            9.68636696200352975694158885744400e-01 1.88297152596150670902583335042380e-04 3.11750066470508728488919558685666e-02 3.39348451645763761214819043132707e-05
            9.24103291705213436557642125990242e-01 4.55664178577059599765247677183311e-04 7.54410441162094991263131760206306e-02 1.25058552118981685741597176253492e-04
            8.62208865680085256855136321973987e-01 8.27262281141620233262945482977102e-04 1.36963872038773110118015097214084e-01 2.95717055589187307381576763631870e-04
            7.85486086304269637281549876206554e-01 1.28788597652830711683125120003979e-03 2.13226027719202043675394975252857e-01 5.48438834010235164335977930960553e-04
            6.97075673538781748206361044140067e-01 1.81867919556982456964766559082136e-03 3.01105647265648423971384772812598e-01 8.67129106424570556856112890642407e-04
            6.00597046998717187804572859022301e-01 2.39791188036389242912993502443442e-03 3.97005041120918944486106738622766e-01 1.21866058435084363698042153600909e-03
            5.00000000000000000000000000000000e-01 3.00187049487862745930644159386702e-03 4.96998129505121399862588305040845e-01 1.55748442427121281880775249817361e-03
            3.99402953001282645661973447204218e-01 3.60582910939336465788729313430849e-03 5.96991217889324077283674796490232e-01 1.83254515960601925030071246425223e-03
            3.02924326461218529349395112149068e-01 4.18506179418742969844391410561002e-03 6.92890611744594098198035680979956e-01 1.99539803543435185703702750004140e-03
            2.14513913695730473740752586309100e-01 4.71585501322894801862206648479514e-03 7.80770231291040617271903556684265e-01 2.00821972748592078583662612345506e-03
            1.37791134319915020700619834315148e-01 5.17647870861563490219037220185783e-03 8.57032386971469439806980972207384e-01 1.85040836132341519892907655275849e-03
            7.58967082947862858866017177206231e-02 5.54807681118019754146208910583482e-03 9.18555214894033467132317127834540e-01 1.52268816745214146876619132342512e-03
            3.13633037996470243058411142555997e-02 5.81544383716110519638720077750804e-03 9.62821252363191826262323047558311e-01 1.04805719819146326600434182552135e-03
            6.00374098975725578597462117613759e-03 5.96769608388516420438563159223122e-03 9.88028562926357545315170227695489e-01 4.70041891913291847403255641069109e-04
            ];
        
    case  29
        
        % ALG. DEG.:   29
        % PTS CARD.:  256
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.1e-15
        
        xyw_bar=[
            9.94700467495824969255124869960127e-01 5.27144745941222295920747242803373e-03 2.80850447628077856676576118388766e-05 1.95355620602029023035665737662026e-06
            9.72287511536616300134028278989717e-01 2.75656252300003658251004168278087e-02 1.46863233383334040871304182473978e-04 2.34217123568435754352168309821991e-05
            9.32815601193915933109224170038942e-01 6.68283529008377730207257627625950e-02 3.56045905246293870050067198462784e-04 8.67951091524684205467249364929216e-05
            8.77702204177501554838158881466370e-01 1.21649674678348099221558697990986e-01 6.48121144150345940282420542644104e-04 2.06926836109061045845466941806023e-04
            8.08938122201322107329701793787535e-01 1.90049339166975139736592836925411e-01 1.01253863170275293370536928705405e-03 3.88037026569801085824906738963591e-04
            7.29008388828613629328856404754333e-01 2.69555482319624839515626035790774e-01 1.43612885176153115551755945489276e-03 6.22334338912883429373312793586592e-04
            6.40801775389629346690867350844201e-01 3.57294641943606106071484873609734e-01 1.90358266676454723764777554606553e-03 8.90476072747813378500947312232938e-04
            5.47506254918818435761806995287770e-01 4.50095739771187941524743791887886e-01 2.39800530999362271344921282434370e-03 1.16382522716153053062015931828910e-03
            4.52493745081181064637831923391786e-01 5.44604727724637527330742159392685e-01 2.90152719418146354257714847335592e-03 1.40819977829512795082511633637523e-03
            3.59198224610370542286830186640145e-01 6.37405825552219029717093690123875e-01 3.39594983741042799607612323598005e-03 1.58858983497968094325547561140866e-03
            2.70991611171386148626538670214359e-01 7.25144985176200296272952527942834e-01 3.86340365241355510050880184280686e-03 1.67417342464032631137171858171087e-03
            1.91061877798678114714903131243773e-01 8.04651128328849662985078339261236e-01 4.28699387247222230001852949499153e-03 1.64291248068185722665257575414444e-03
            1.22297795822498389650689887275803e-01 8.73050792817476772889051517267944e-01 4.65141136002478194910736419842578e-03 1.48506470566322264920122719900064e-03
            6.71843988060842889353807549923658e-02 9.27872114594986974189794182166224e-01 4.94348659892873687482506284140982e-03 1.20509870391844581792584190083062e-03
            2.77124884633839219105766460415907e-02 9.67134842265824357099290864425711e-01 5.15266927079172099013248953269795e-03 8.21746428634548579592045136621437e-04
            5.29953250417514176717759255552664e-03 9.89429020036412643079870576912072e-01 5.27144745941221515295183053240180e-03 3.66674469847457103460069971490043e-04
            9.94700467495824969255124869960127e-01 5.15266927079175308251679510362919e-03 1.46863233383277662358334936243409e-04 4.47899603467711012274622695006343e-06
            9.72287511536616300134028278989717e-01 2.69445064465504702244658119525411e-02 7.67982016833229641505909057741519e-04 5.36998917401819832271453458716337e-05
            9.32815601193915933109224170038942e-01 6.53225519292510992119815682599437e-02 1.86184687683296767879426170111401e-03 1.98998599848442382729413191455592e-04
            8.77702204177501554838158881466370e-01 1.18908619566670148426368314176216e-01 3.38917625582829673547280435741413e-03 4.74429389614981127185761033970834e-04
            8.08938122201322107329701793787535e-01 1.85767077714389661213800764016923e-01 5.29480008428823145649744219554123e-03 8.89667928651336620271550259531068e-04
            7.29008388828613629328856404754333e-01 2.63481759273125559772665837954264e-01 7.50985189826081089847775729140267e-03 1.42685069804705387704057972797500e-03
            6.40801775389629346690867350844201e-01 3.49243947954787914422780659151613e-01 9.95427665558273888635199000418652e-03 2.04162991907839508690103436094887e-03
            5.47506254918818435761806995287770e-01 4.39954017390866036762986368557904e-01 1.25397276903155274752066361543257e-02 2.66834840044501923200792603552145e-03
            4.52493745081181064637831923391786e-01 5.32333494145750818482554223010084e-01 1.51727607730681723907650848559570e-02 3.22863565613302518075844460554435e-03
            3.59198224610370542286830186640145e-01 6.23043563581828552244701313611586e-01 1.77582118078009054684684997482691e-02 3.64222311581059937993232189512582e-03
            2.70991611171386148626538670214359e-01 7.08805752263490962405967366066761e-01 2.02026365651228889674939637188800e-02 3.83844401672051808299679365177326e-03
            1.91061877798678114714903131243773e-01 7.86520433822226472386773821199313e-01 2.24176883790954128983230475569144e-02 3.76677080680787139399523510974177e-03
            1.22297795822498389650689887275803e-01 8.53378891969946096196508733555675e-01 2.43233122075554586416501479106955e-02 3.40486693313773926966581129249789e-03
            6.71843988060842889353807549923658e-02 9.06964959607365006633017401327379e-01 2.58506415865507044316018436802551e-02 2.76297774264764716550812906348256e-03
            2.77124884633839219105766460415907e-02 9.45343005090065524598230695119128e-01 2.69445064465505534911926588392816e-02 1.88405072964969587563655029072152e-03
            5.29953250417514176717759255552664e-03 9.67134842265824468121593326941365e-01 2.75656252300003901112290805031080e-02 8.40689144956720062727695275128781e-04
            9.94700467495824969255124869960127e-01 4.94348659892872993593115893418144e-03 3.56045905246300808943971105691162e-04 6.84643325431189101208315217483857e-06
            9.72287511536616300134028278989717e-01 2.58506415865506732065792760977274e-02 1.86184687683302665939244491255522e-03 8.20837352202376843279124640773148e-05
            9.32815601193915933109224170038942e-01 6.26706553631491136702180710926768e-02 4.51374344293495322055775886838092e-03 3.04182147297234369009988341758799e-04
            8.77702204177501554838158881466370e-01 1.14081291934854611724858841625974e-01 8.21650388764383343698227690765634e-03 7.25195808331868222058880224523136e-04
            8.08938122201322107329701793787535e-01 1.78225500404012277000376229807443e-01 1.28363773946656156699219764050213e-02 1.35991459801602454851576773364741e-03
            7.29008388828613629328856404754333e-01 2.52785202693344701607713886915008e-01 1.82064084780416690634297083306592e-02 2.18103297980519863011328318691540e-03
            6.40801775389629346690867350844201e-01 3.35065707837710224925586999233929e-01 2.41325167726604283835456499218708e-02 3.12076252418117677800424303313775e-03
            5.47506254918818435761806995287770e-01 4.22093224854388937572480244853068e-01 3.04005202267926266657127598591615e-02 4.07874199518323734409142033996432e-03
            4.52493745081181064637831923391786e-01 5.10722376339527550648256237764144e-01 3.67838785792914402250630701018963e-02 4.93517706893878849855328283524614e-03
            3.59198224610370542286830186640145e-01 5.97749893356205874717090864578495e-01 4.30518820334235829960789487813599e-02 5.56737207772652325371565140699204e-03
            2.70991611171386148626538670214359e-01 6.80030398500571453546115208155243e-01 4.89779903280423978273461216303986e-02 5.86730833370437653007112288605640e-03
            1.91061877798678114714903131243773e-01 7.54590100789903517330969862086931e-01 5.43480214114183679541270066692960e-02 5.75775122671207378183266811788599e-03
            1.22297795822498389650689887275803e-01 8.18734309259061210362062865897315e-01 5.89678949184403444760960155690555e-02 5.20455790026651786417399847550769e-03
            6.71843988060842889353807549923658e-02 8.70144945830766625149976789543871e-01 6.26706553631490859146424554637633e-02 4.22338902551632790205449197173948e-03
            2.77124884633839219105766460415907e-02 9.06964959607365006633017401327379e-01 6.53225519292510714564059526310302e-02 2.87989260727580667978409145746355e-03
            5.29953250417514176717759255552664e-03 9.27872114594987085212096644681878e-01 6.68283529008377730207257627625950e-02 1.28504737981659038735526845442791e-03
            9.94700467495824969255124869960127e-01 4.65141136002473858102046477824842e-03 6.48121144150292163854665261624177e-04 8.96676417241144234750258729116368e-06
            9.72287511536616300134028278989717e-01 2.43233122075554065999458686064827e-02 3.38917625582829326602585240379995e-03 1.07504954590331375046362294423119e-04
            9.32815601193915933109224170038942e-01 5.89678949184402889649447843112284e-02 8.21650388764377792583104564982932e-03 3.98386937980330102255371871322609e-04
            8.77702204177501554838158881466370e-01 1.07341044959456871632497154678276e-01 1.49567508630415735293439638553536e-02 9.49787948058614258338427926986469e-04
            8.08938122201322107329701793787535e-01 1.67695431278192086388401094154688e-01 2.33664465204858062818971120577771e-02 1.78107826982021387289545355514520e-03
            7.29008388828613629328856404754333e-01 2.37849934438738280784164658143709e-01 3.31416767326480898869789371019579e-02 2.85649587978501449947454915445633e-03
            6.40801775389629346690867350844201e-01 3.15269073477167649510732871931395e-01 4.39291511332030037983997772244038e-02 4.08725836548662173342005843323932e-03
            5.47506254918818435761806995287770e-01 3.97154757434285576866983547006384e-01 5.53389876468959873712094577058451e-02 5.34192275487170315645402141058184e-03
            4.52493745081181064637831923391786e-01 4.80547446743216477571536415780429e-01 6.69588081756025133017828920856118e-02 6.46359458750240915836382527004389e-03
            3.59198224610370542286830186640145e-01 5.62433130700334071860879703308456e-01 7.83686446892953858522901100513991e-02 7.29157951691145837097574045060355e-03
            2.70991611171386148626538670214359e-01 6.39852269738763523854174763982883e-01 8.91561190898503275192865658027586e-02 7.68440561689799107869536243242692e-03
            1.91061877798678114714903131243773e-01 7.10006772899309357427455324796028e-01 9.89313493020125278576415439601988e-02 7.54091882526193232894051732273510e-03
            1.22297795822498389650689887275803e-01 7.70361159218044599938934879901353e-01 1.07341044959456954899224001565017e-01 6.81640229004776450338898285963296e-03
            6.71843988060842889353807549923658e-02 8.18734309259061099339760403381661e-01 1.14081291934854611724858841625974e-01 5.53136677061809214211995922028109e-03
            2.77124884633839219105766460415907e-02 8.53378891969945874151903808524366e-01 1.18908619566670203937519545434043e-01 3.77179136816235306695066498150481e-03
            5.29953250417514176717759255552664e-03 8.73050792817476772889051517267944e-01 1.21649674678348085343770890176529e-01 1.68302477759986663687075125750425e-03
            9.94700467495824969255124869960127e-01 4.28699387247221969793331552978088e-03 1.01253863170281104694181451009172e-03 1.07630829280184121989191389778817e-05
            9.72287511536616300134028278989717e-01 2.24176883790953677955126721599299e-02 5.29480008428833207045904885035270e-03 1.29041504736865868903025744707236e-04
            9.32815601193915933109224170038942e-01 5.43480214114183887708087183909811e-02 1.28363773946656781199671115700767e-02 4.78196099336967214016302918366819e-04
            8.77702204177501554838158881466370e-01 9.89313493020125417354293517746555e-02 2.33664465204859034264117667589744e-02 1.14005969739227794833291707021772e-03
            8.08938122201322107329701793787535e-01 1.54557236650721019355358976099524e-01 3.65046411479568733149392301129410e-02 2.13788304797260346468146252618681e-03
            7.29008388828613629328856404754333e-01 2.19215445073292103383977291741758e-01 5.17761660980942672871663035039091e-02 3.42873989395896120249696004123052e-03
            6.40801775389629346690867350844201e-01 2.90569137314362013757573777184007e-01 6.86290872960086395515588719717925e-02 4.90606197398618659732960622932296e-03
            5.47506254918818435761806995287770e-01 3.66039440453814757248096611874644e-01 8.64543046273668069900963928375859e-02 6.41207424442504989037239226945530e-03
            4.52493745081181064637831923391786e-01 4.42898681747507794170815031975508e-01 1.04607573171311196702504275890533e-01 7.75845145703997398495266324403019e-03
            3.59198224610370542286830186640145e-01 5.18368984886960260105581710377010e-01 1.22432790502669197607588102982845e-01 8.75230724347830779663581779459491e-03
            2.70991611171386148626538670214359e-01 5.89722677128030170479178195819259e-01 1.39285711700583680894283133966383e-01 9.22382849787392426754717433823316e-03
            1.91061877798678114714903131243773e-01 6.54380885550600921440889123914531e-01 1.54557236650720963844207744841697e-01 9.05159688703198635206792488361316e-03
            1.22297795822498389650689887275803e-01 7.10006772899309468449757787311682e-01 1.67695431278192086388401094154688e-01 8.18193739768985350835350800480228e-03
            6.71843988060842889353807549923658e-02 7.54590100789903517330969862086931e-01 1.78225500404012193733649382920703e-01 6.63946972539116320871643850409782e-03
            2.77124884633839219105766460415907e-02 7.86520433822226472386773821199313e-01 1.85767077714389605702649532759096e-01 4.52739723072229181127790198502225e-03
            5.29953250417514176717759255552664e-03 8.04651128328849774007380801776890e-01 1.90049339166975084225441605667584e-01 2.02018642432363025596142591666649e-03
            9.94700467495824969255124869960127e-01 3.86340365241350739561321248061176e-03 1.43612885176152334926191755926084e-03 1.21704175390405777148431637479931e-05
            9.72287511536616300134028278989717e-01 2.02026365651228542730244441827381e-02 7.50985189826084559294727682754456e-03 1.45914419039311658377391922591926e-04
            9.32815601193915933109224170038942e-01 4.89779903280423770106644099087134e-02 1.82064084780416898801114200523443e-02 5.40722972534311428945141830837429e-04
            8.77702204177501554838158881466370e-01 8.91561190898502997637109501738450e-02 3.31416767326481453981301683597849e-02 1.28912901902640975732028394418194e-03
            8.08938122201322107329701793787535e-01 1.39285711700583680894283133966383e-01 5.17761660980942117760150722460821e-02 2.41742347591979763604674857901955e-03
            7.29008388828613629328856404754333e-01 1.97555157846122514397890768123034e-01 7.34364533252638562732528271226329e-02 3.87706723262505039903635939424476e-03
            6.40801775389629346690867350844201e-01 2.61858518993304811850464375311276e-01 9.73397056170658414586682738445234e-02 5.54755761849495473980731219398876e-03
            5.47506254918818435761806995287770e-01 3.29871736056657594016172652118257e-01 1.22622009024523970222020352593972e-01 7.25048960115635531681332892617320e-03
            4.52493745081181064637831923391786e-01 3.99136652771956423890742371440865e-01 1.48369602146862566982576936425176e-01 8.77291332976271390209355161005078e-03
            3.59198224610370542286830186640145e-01 4.67149869835308928500694491958711e-01 1.73651905554320529212475321401143e-01 9.89672144082527453778919124260938e-03
            2.70991611171386148626538670214359e-01 5.31453230982491309219994946033694e-01 1.97555157846122542153466383751947e-01 1.04298967942909708017662495649347e-02
            1.91061877798678114714903131243773e-01 5.89722677128029837412270808272297e-01 2.19215445073292047872826060483931e-01 1.02351449159131420441060100756658e-02
            1.22297795822498389650689887275803e-01 6.39852269738763301809569838951575e-01 2.37849934438738253028589042514795e-01 9.25177247765662444378698126001836e-03
            6.71843988060842889353807549923658e-02 6.80030398500571120479207820608281e-01 2.52785202693344590585411424399354e-01 7.50761834097533539700286553397746e-03
            2.77124884633839219105766460415907e-02 7.08805752263490518316757516004145e-01 2.63481759273125559772665837954264e-01 5.11938029572822639323703697300516e-03
            5.29953250417514176717759255552664e-03 7.25144985176200074228347602911526e-01 2.69555482319624784004474804532947e-01 2.28433734601416827195152770002551e-03
            9.94700467495824969255124869960127e-01 3.39594983741040804675614950269846e-03 1.90358266676462269811898053717414e-03 1.31378903579940360359578840898642e-05
            9.72287511536616300134028278989717e-01 1.77582118078008603656581243512846e-02 9.95427665558283950031359665899799e-03 1.57513711656929398510426643120752e-04
            9.32815601193915933109224170038942e-01 4.30518820334235621793972370596748e-02 2.41325167726605047113785929013829e-02 5.83707099975505562701461048646934e-04
            8.77702204177501554838158881466370e-01 7.83686446892953580967144944224856e-02 4.39291511332030870651266241111443e-02 1.39160679203883066480662566988258e-03
            8.08938122201322107329701793787535e-01 1.22432790502669239240951526426215e-01 6.86290872960086534293466797862493e-02 2.60959367035642694765407156864967e-03
            7.29008388828613629328856404754333e-01 1.73651905554320501456899705772230e-01 9.73397056170658692142438894734369e-02 4.18527006566569385187515095481103e-03
            6.40801775389629346690867350844201e-01 2.30174860047128393425452941301046e-01 1.29023364563242259883679707854753e-01 5.98855409131565218777693004881257e-03
            5.47506254918818435761806995287770e-01 2.89958795200723484786209382946254e-01 1.62534949880458079451983621765976e-01 7.82685861978054710219598888443215e-03
            4.52493745081181064637831923391786e-01 3.50842980188906194971565355444909e-01 1.96663274729912795901753952421132e-01 9.47030560594003430197052040284689e-03
            3.59198224610370542286830186640145e-01 4.10626915342501064287716872058809e-01 2.30174860047128393425452941301046e-01 1.06834495017186694582056460944841e-02
            2.70991611171386148626538670214359e-01 4.67149869835308984011845723216538e-01 2.61858518993304867361615606569103e-01 1.12590089936544929249961910500133e-02
            1.91061877798678114714903131243773e-01 5.18368984886960038060976785345702e-01 2.90569137314361847224120083410526e-01 1.10487755471080899977165401537604e-02
            1.22297795822498389650689887275803e-01 5.62433130700333960838577240792802e-01 3.15269073477167593999581640673568e-01 9.98723109035925052379134569946473e-03
            6.71843988060842889353807549923658e-02 5.97749893356205652672485939547187e-01 3.35065707837710058392133305460447e-01 8.10442750193224661914292994424613e-03
            2.77124884633839219105766460415907e-02 6.23043563581828330200096388580278e-01 3.49243947954787747889326965378132e-01 5.52633932323201310793381324515394e-03
            5.29953250417514176717759255552664e-03 6.37405825552218918694791227608221e-01 3.57294641943605939538031179836253e-01 2.46592801736949217283889623786308e-03
            9.94700467495824969255124869960127e-01 2.90152719418141843976677307637146e-03 2.39800530999361230510835696350114e-03 1.36305301179685297289405701559595e-05
            9.72287511536616300134028278989717e-01 1.51727607730681411657425172734293e-02 1.25397276903155587002292037368534e-02 1.63420102636676424284409292475573e-04
            9.32815601193915933109224170038942e-01 3.67838785792914263472752622874395e-02 3.04005202267926405435005676736182e-02 6.05594733209738680934641763542459e-04
            8.77702204177501554838158881466370e-01 6.69588081756024161572682373844145e-02 5.53389876468960290045728811492154e-02 1.44378874951663397317724868429423e-03
            8.08938122201322107329701793787535e-01 1.04607573171311099557989621189336e-01 8.64543046273667931123085850231291e-02 2.70744724991633760324827306931184e-03
            7.29008388828613629328856404754333e-01 1.48369602146862400449123242651694e-01 1.22622009024523970222020352593972e-01 4.34220777669807548454494749989863e-03
            6.40801775389629346690867350844201e-01 1.96663274729912629368300258647650e-01 1.62534949880458023940832390508149e-01 6.21311068067285487276363298292381e-03
            5.47506254918818435761806995287770e-01 2.47743155743588239214858504055883e-01 2.04750589337593325023334500656347e-01 8.12034727334179709523631629508600e-03
            4.52493745081181064637831923391786e-01 2.99763099175230474102704647521023e-01 2.47743155743588516770614660345018e-01 9.82541962755741071777748629756388e-03
            3.59198224610370542286830186640145e-01 3.50842980188905917415809199155774e-01 2.89958795200723540297360614204081e-01 1.10840535450477326029350422231801e-02
            2.70991611171386148626538670214359e-01 3.99136652771956146334986215151730e-01 3.29871736056657705038475114633911e-01 1.16811951542209538357219855697622e-02
            1.91061877798678114714903131243773e-01 4.42898681747507239059302719397238e-01 3.66039440453814646225794149358990e-01 1.14630784515487131602906600846836e-02
            1.22297795822498389650689887275803e-01 4.80547446743215977971175334459986e-01 3.97154757434285576866983547006384e-01 1.03617285928574824616088534412484e-02
            6.71843988060842889353807549923658e-02 5.10722376339526884514441462670220e-01 4.22093224854388826550177782337414e-01 8.40832433091234614230824462310920e-03
            2.77124884633839219105766460415907e-02 5.32333494145750152348739447916159e-01 4.39954017390865925740683906042250e-01 5.73356395394131624493327947789112e-03
            5.29953250417514176717759255552664e-03 5.44604727724636972219229846814414e-01 4.50095739771187886013592560630059e-01 2.55839447533868233689013926834832e-03
            9.94700467495824969255124869960127e-01 2.39800530999360970302314299829050e-03 2.90152719418142104185198704158211e-03 1.36305301179684992357544690011473e-05
            9.72287511536616300134028278989717e-01 1.25397276903154893112901646645696e-02 1.51727607730682105546815563457130e-02 1.63420102636676071918703234686632e-04
            9.32815601193915933109224170038942e-01 3.04005202267926093184780000910905e-02 3.67838785792914541028508779163531e-02 6.05594733209737271471817532386694e-04
            8.77702204177501554838158881466370e-01 5.53389876468959179822704186335613e-02 6.69588081756025271795706999000686e-02 1.44378874951663072057073122778093e-03
            8.08938122201322107329701793787535e-01 8.64543046273667514789451615797589e-02 1.04607573171311141191353044632706e-01 2.70744724991633153171610715048701e-03
            7.29008388828613629328856404754333e-01 1.22622009024523845321930082263862e-01 1.48369602146862511471425705167348e-01 4.34220777669806594356582962745961e-03
            6.40801775389629346690867350844201e-01 1.62534949880457912918529927992495e-01 1.96663274729912740390602721163305e-01 6.21311068067284099497582516846705e-03
            5.47506254918818435761806995287770e-01 2.04750589337593102978729575625039e-01 2.47743155743588461259463429087191e-01 8.12034727334177801327808055020796e-03
            4.52493745081181064637831923391786e-01 2.47743155743588211459282888426969e-01 2.99763099175230807169612035067985e-01 9.82541962755738816637229859907166e-03
            3.59198224610370542286830186640145e-01 2.89958795200723262741604457914946e-01 3.50842980188906194971565355444909e-01 1.10840535450477083168063785478807e-02
            2.70991611171386148626538670214359e-01 3.29871736056657316460416495829122e-01 3.99136652771956534913044833956519e-01 1.16811951542209278148698459176558e-02
            1.91061877798678114714903131243773e-01 3.66039440453814257647735530554201e-01 4.42898681747507627637361338202027e-01 1.14630784515486871394385204325772e-02
            1.22297795822498389650689887275803e-01 3.97154757434285132777773696943768e-01 4.80547446743216422060385184522602e-01 1.03617285928574599102036657427561e-02
            6.71843988060842889353807549923658e-02 4.22093224854388382460967932274798e-01 5.10722376339527328603651312732836e-01 8.40832433091232706035000887823116e-03
            2.77124884633839219105766460415907e-02 4.39954017390865426140322824721807e-01 5.32333494145750707460251760494430e-01 5.73356395394130323450720965183791e-03
            5.29953250417514176717759255552664e-03 4.50095739771187386413231479309616e-01 5.44604727724637527330742159392685e-01 2.55839447533867713271971133792704e-03
            9.94700467495824969255124869960127e-01 1.90358266676462226443811154297236e-03 3.39594983741040848043701849690024e-03 1.31378903579940258715625170382602e-05
            9.72287511536616300134028278989717e-01 9.95427665558278051971541344755678e-03 1.77582118078009193462563075627259e-02 1.57513711656929290090209394570309e-04
            9.32815601193915933109224170038942e-01 2.41325167726604943030377370405404e-02 4.30518820334235691182911409669032e-02 5.83707099975505020600374805894717e-04
            8.77702204177501554838158881466370e-01 4.39291511332030107372936811316322e-02 7.83686446892954413634413413092261e-02 1.39160679203882958060445318437814e-03
            8.08938122201322107329701793787535e-01 6.86290872960086950627101032296196e-02 1.22432790502669197607588102982845e-01 2.60959367035642477924972659764080e-03
            7.29008388828613629328856404754333e-01 9.73397056170658414586682738445234e-02 1.73651905554320529212475321401143e-01 4.18527006566569038242819900119684e-03
            6.40801775389629346690867350844201e-01 1.29023364563242259883679707854753e-01 2.30174860047128393425452941301046e-01 5.98855409131564785096824010679484e-03
            5.47506254918818435761806995287770e-01 1.62534949880457996185256774879235e-01 2.89958795200723540297360614204081e-01 7.82685861978054016330208497720378e-03
            4.52493745081181064637831923391786e-01 1.96663274729912740390602721163305e-01 3.50842980188906250482716586702736e-01 9.47030560594002736307661649561851e-03
            3.59198224610370542286830186640145e-01 2.30174860047128337914301710043219e-01 4.10626915342501119798868103316636e-01 1.06834495017186625193117421872557e-02
            2.70991611171386148626538670214359e-01 2.61858518993304756339313144053449e-01 4.67149869835309095034148185732192e-01 1.12590089936544842513788111659778e-02
            1.91061877798678114714903131243773e-01 2.90569137314361791712968852152699e-01 5.18368984886960149083279247861356e-01 1.10487755471080813240991602697250e-02
            1.22297795822498389650689887275803e-01 3.15269073477167482977279178157914e-01 5.62433130700334071860879703308456e-01 9.98723109035924185017396581542926e-03
            6.71843988060842889353807549923658e-02 3.35065707837709947369830842944793e-01 5.97749893356205763694788402062841e-01 8.10442750193224141497250201382485e-03
            2.77124884633839219105766460415907e-02 3.49243947954787636867024502862478e-01 6.23043563581828441222398851095932e-01 5.52633932323200877112512330313621e-03
            5.29953250417514176717759255552664e-03 3.57294641943605884026879948578426e-01 6.37405825552219029717093690123875e-01 2.46592801736949043811542026105599e-03
            9.94700467495824969255124869960127e-01 1.43612885176152226505974507375640e-03 3.86340365241350826297495046901531e-03 1.21704175390405218106686449641707e-05
            9.72287511536616300134028278989717e-01 7.50985189826078487762561763929625e-03 2.02026365651229167230695793477935e-02 1.45914419039310980751034119151655e-04
            9.32815601193915933109224170038942e-01 1.82064084780416725328766602842734e-02 4.89779903280423978273461216303986e-02 5.40722972534308935280145114177230e-04
            8.77702204177501554838158881466370e-01 3.31416767326480551925094175658160e-02 8.91561190898503830304377970605856e-02 1.28912901902640390262855252245799e-03
            8.08938122201322107329701793787535e-01 5.17761660980942325926967839677673e-02 1.39285711700583653138707518337469e-01 2.41742347591978636034415472977344e-03
            7.29008388828613629328856404754333e-01 7.34364533252638007621015958648059e-02 1.97555157846122569909041999380861e-01 3.87706723262503261812073063197204e-03
            6.40801775389629346690867350844201e-01 9.73397056170657998253048504011531e-02 2.61858518993304867361615606569103e-01 5.54755761849492958631691053028590e-03
            5.47506254918818435761806995287770e-01 1.22622009024523873077505697892775e-01 3.29871736056657705038475114633911e-01 7.25048960115632148970554737843486e-03
            4.52493745081181064637831923391786e-01 1.48369602146862428204698858280608e-01 3.99136652771956534913044833956519e-01 8.77291332976267400345360414348761e-03
            3.59198224610370542286830186640145e-01 1.73651905554320390434597243256576e-01 4.67149869835309039522996954474365e-01 9.89672144082522943497881584562492e-03
            2.70991611171386148626538670214359e-01 1.97555157846122403375588305607380e-01 5.31453230982491420242297408549348e-01 1.04298967942909222295089222143361e-02
            1.91061877798678114714903131243773e-01 2.19215445073291881339372366710450e-01 5.89722677128029948434573270787951e-01 1.02351449159130952065721587018743e-02
            1.22297795822498389650689887275803e-01 2.37849934438738086495135348741314e-01 6.39852269738763412831872301467229e-01 9.25177247765658281042355781664810e-03
            6.71843988060842889353807549923658e-02 2.52785202693344424051957730625873e-01 6.80030398500571342523812745639589e-01 7.50761834097530070253334599783557e-03
            2.77124884633839219105766460415907e-02 2.63481759273125282216909681665129e-01 7.08805752263490740361362441035453e-01 5.11938029572820297447011128610939e-03
            5.29953250417514176717759255552664e-03 2.69555482319624617471021110759466e-01 7.25144985176200185250650065427180e-01 2.28433734601415786361067183918294e-03
            9.94700467495824969255124869960127e-01 1.01253863170281234798442149269704e-03 4.28699387247221883057157754137734e-03 1.07630829280184291395780840638885e-05
            9.72287511536616300134028278989717e-01 5.29480008428828263083998351135051e-03 2.24176883790954163677699995105286e-02 1.29041504736866085743460241808123e-04
            9.32815601193915933109224170038942e-01 1.28363773946656902630314434077263e-02 5.43480214114183748930209105765243e-02 4.78196099336968027167932282495144e-04
            8.77702204177501554838158881466370e-01 2.33664465204858756708361511300609e-02 9.89313493020125694910049674035690e-02 1.14005969739227989989682754412570e-03
            8.08938122201322107329701793787535e-01 3.65046411479569773983477887213667e-02 1.54557236650720908333056513583870e-01 2.13788304797260693412841447980099e-03
            7.29008388828613629328856404754333e-01 5.17761660980943158594236308545078e-02 2.19215445073292047872826060483931e-01 3.42873989395896684034825696585358e-03
            6.40801775389629346690867350844201e-01 6.86290872960087783294369501163601e-02 2.90569137314361847224120083410526e-01 4.90606197398619527094698611335843e-03
            5.47506254918818435761806995287770e-01 8.64543046273669180123988553532399e-02 3.66039440453814646225794149358990e-01 6.41207424442506029871324813029787e-03
            4.52493745081181064637831923391786e-01 1.04607573171311307724806738406187e-01 4.42898681747507683148512569459854e-01 7.75845145703998699537873307008340e-03
            3.59198224610370542286830186640145e-01 1.22432790502669364141041796756326e-01 5.18368984886960149083279247861356e-01 8.75230724347832340914710158585876e-03
            2.70991611171386148626538670214359e-01 1.39285711700583847427736827739864e-01 5.89722677128029948434573270787951e-01 9.22382849787393988005845812949701e-03
            1.91061877798678114714903131243773e-01 1.54557236650721102622085822986264e-01 6.54380885550600810418586661398876e-01 9.05159688703200196457920867487701e-03
            1.22297795822498389650689887275803e-01 1.67695431278192225166279172299255e-01 7.10006772899309357427455324796028e-01 8.18193739768986738614131581925903e-03
            6.71843988060842889353807549923658e-02 1.78225500404012388022678692323097e-01 7.54590100789903295286364937055623e-01 6.63946972539117448441903235334394e-03
            2.77124884633839219105766460415907e-02 1.85767077714389772236103226532578e-01 7.86520433822226250342168896168005e-01 4.52739723072229961753354388065418e-03
            5.29953250417514176717759255552664e-03 1.90049339166975278514470915069978e-01 8.04651128328849551962775876745582e-01 2.02018642432363372540837787028067e-03
            9.94700467495824969255124869960127e-01 6.48121144150291513333361770321517e-04 4.65141136002473944838220276665197e-03 8.96676417241144065343669278256300e-06
            9.72287511536616300134028278989717e-01 3.38917625582823515278940718076228e-03 2.43233122075554655805440518179239e-02 1.07504954590331361493835138354314e-04
            9.32815601193915933109224170038942e-01 8.21650388764377445638409369621513e-03 5.89678949184402889649447843112284e-02 3.98386937980330048045263247047387e-04
            8.77702204177501554838158881466370e-01 1.49567508630415041404049247830699e-02 1.07341044959456941021436193750560e-01 9.49787948058614041497993429885582e-04
            8.08938122201322107329701793787535e-01 2.33664465204858548541544394083758e-02 1.67695431278192030877249862896861e-01 1.78107826982021343921458456094342e-03
            7.29008388828613629328856404754333e-01 3.31416767326480829480850331947295e-02 2.37849934438738280784164658143709e-01 2.85649587978501363211281116605278e-03
            6.40801775389629346690867350844201e-01 4.39291511332030315539753928533173e-02 3.15269073477167593999581640673568e-01 4.08725836548662086605832044483577e-03
            5.47506254918818435761806995287770e-01 5.53389876468959804323155537986167e-02 3.97154757434285576866983547006384e-01 5.34192275487170142173054543377475e-03
            4.52493745081181064637831923391786e-01 6.69588081756024855462072764566983e-02 4.80547446743216477571536415780429e-01 6.46359458750240829100208728164034e-03
            3.59198224610370542286830186640145e-01 7.83686446892953719745023022369423e-02 5.62433130700334071860879703308456e-01 7.29157951691145663625226447379646e-03
            2.70991611171386148626538670214359e-01 8.91561190898503275192865658027586e-02 6.39852269738763523854174763982883e-01 7.68440561689798934397188645561982e-03
            1.91061877798678114714903131243773e-01 9.89313493020125139798537361457420e-02 7.10006772899309357427455324796028e-01 7.54091882526193059421704134592801e-03
            1.22297795822498389650689887275803e-01 1.07341044959456871632497154678276e-01 7.70361159218044710961237342417007e-01 6.81640229004776276866550688282587e-03
            6.71843988060842889353807549923658e-02 1.14081291934854583969283225997060e-01 8.18734309259061099339760403381661e-01 5.53136677061809127475822123187754e-03
            2.77124884633839219105766460415907e-02 1.18908619566670120670792698547302e-01 8.53378891969945985174206271040021e-01 3.77179136816235219958892699310127e-03
            5.29953250417514176717759255552664e-03 1.21649674678348085343770890176529e-01 8.73050792817476772889051517267944e-01 1.68302477759986620318988226330248e-03
            9.94700467495824969255124869960127e-01 3.56045905246301838936034966920374e-04 4.94348659892872906856942094577789e-03 6.84643325431193505779640939845621e-06
            9.72287511536616300134028278989717e-01 1.86184687683297678609251057935126e-03 2.58506415865507217788366034483261e-02 8.20837352202382128764715507607264e-05
            9.32815601193915933109224170038942e-01 4.51374344293496536362209070603058e-03 6.26706553631490997924302632782201e-02 3.04182147297236320573898815666780e-04
            8.77702204177501554838158881466370e-01 8.21650388764380047723623334832155e-03 1.14081291934854639480434457254887e-01 7.25195808331872884128221912192203e-04
            8.08938122201322107329701793787535e-01 1.28363773946657214880540109902540e-02 1.78225500404012165978073767291789e-01 1.35991459801603322213314761768288e-03
            7.29008388828613629328856404754333e-01 1.82064084780417488607096032637855e-02 2.52785202693344646096562655657181e-01 2.18103297980521250790109100137215e-03
            6.40801775389629346690867350844201e-01 2.41325167726605845086584878345093e-02 3.35065707837710058392133305460447e-01 3.12076252418119672732421676641934e-03
            5.47506254918818435761806995287770e-01 3.04005202267927446269091262820439e-02 4.22093224854388826550177782337414e-01 4.07874199518326336494355999207073e-03
            4.52493745081181064637831923391786e-01 3.67838785792915859418350521536922e-02 5.10722376339527439625953775248490e-01 4.93517706893882059093758840617738e-03
            3.59198224610370542286830186640145e-01 4.30518820334237148350631230186991e-02 5.97749893356205763694788402062841e-01 5.56737207772655881554690893153747e-03
            2.70991611171386148626538670214359e-01 4.89779903280425574219059114966512e-02 6.80030398500571342523812745639589e-01 5.86730833370441382662585638740893e-03
            1.91061877798678114714903131243773e-01 5.43480214114185553042624121644621e-02 7.54590100789903295286364937055623e-01 5.75775122671211021102566363083497e-03
            1.22297795822498389650689887275803e-01 5.89678949184404832539740937136230e-02 8.18734309259061099339760403381661e-01 5.20455790026655169128178002324603e-03
            6.71843988060842889353807549923658e-02 6.26706553631493079592473804950714e-02 8.70144945830766403105371864512563e-01 4.22338902551635479026836961224944e-03
            2.77124884633839219105766460415907e-02 6.53225519292512935010108776623383e-02 9.06964959607364784588412476296071e-01 2.87989260727582489438058921393804e-03
            5.29953250417514176717759255552664e-03 6.68283529008379811875428799794463e-02 9.27872114594986863167491719650570e-01 1.28504737981659862729177934426161e-03
            9.94700467495824969255124869960127e-01 1.46863233383278638140290173197400e-04 5.15266927079175221515505711522565e-03 4.47899603467706692406591698074614e-06
            9.72287511536616300134028278989717e-01 7.67982016833179226104888481785338e-04 2.69445064465505222661700912567539e-02 5.36998917401814750073769932914303e-05
            9.32815601193915933109224170038942e-01 1.86184687683298914599727691410180e-03 6.53225519292510714564059526310302e-02 1.98998599848440485375611341822832e-04
            8.77702204177501554838158881466370e-01 3.38917625582826897989718872850062e-03 1.18908619566670176181943929805129e-01 4.74429389614976627746745219127433e-04
            8.08938122201322107329701793787535e-01 5.29480008428833120309731086194915e-03 1.85767077714389550191498301501269e-01 8.89667928651328163494604872596483e-04
            7.29008388828613629328856404754333e-01 7.50985189826086640962898854922969e-03 2.63481759273125504261514606696437e-01 1.42685069804704043293364090772002e-03
            6.40801775389629346690867350844201e-01 9.95427665558288460312397205598245e-03 3.49243947954787747889326965378132e-01 2.04162991907837557126192962186906e-03
            5.47506254918818435761806995287770e-01 1.25397276903156298238917187859442e-02 4.39954017390865925740683906042250e-01 2.66834840044499407851752437181858e-03
            4.52493745081181064637831923391786e-01 1.51727607730683076991962110469103e-02 5.32333494145750707460251760494430e-01 3.22863565613299438941674601721843e-03
            3.59198224610370542286830186640145e-01 1.77582118078010407768996259392225e-02 6.23043563581828441222398851095932e-01 3.64222311581056468546280235898394e-03
            2.70991611171386148626538670214359e-01 2.02026365651230624398415613995894e-02 7.08805752263490740361362441035453e-01 3.83844401672048165380379813882428e-03
            1.91061877798678114714903131243773e-01 2.24176883790955863706706452376238e-02 7.86520433822226250342168896168005e-01 3.76677080680783539848310859099456e-03
            1.22297795822498389650689887275803e-01 2.43233122075556529306794573130901e-02 8.53378891969945874151903808524366e-01 3.40486693313770674360063672736487e-03
            6.71843988060842889353807549923658e-02 2.58506415865509264762067687115632e-02 9.06964959607364784588412476296071e-01 2.76297774264762114465598941137614e-03
            2.77124884633839219105766460415907e-02 2.69445064465507339024341604272195e-02 9.45343005090065302553625770087820e-01 1.88405072964967809472092152844880e-03
            5.29953250417514176717759255552664e-03 2.75656252300006399114096211633296e-02 9.67134842265824246076988401910057e-01 8.40689144956712039631618882395969e-04
            9.94700467495824969255124869960127e-01 2.80850447628082600061080742470665e-05 5.27144745941222209184573443963018e-03 1.95355620602028345409307934221754e-06
            9.72287511536616300134028278989717e-01 1.46863233383280264443548901454051e-04 2.75656252300004178668046961320215e-02 2.34217123568434941200538945693665e-05
            9.32815601193915933109224170038942e-01 3.56045905246308073098526758570870e-04 6.68283529008377591429379549481382e-02 8.67951091524681223911275029792023e-05
            8.77702204177501554838158881466370e-01 6.48121144150305065860517839126942e-04 1.21649674678348140854922121434356e-01 2.06926836109060314009000514090530e-04
            8.08938122201322107329701793787535e-01 1.01253863170283273098526422018040e-03 1.90049339166975056469865990038670e-01 3.88037026569799784782299756358270e-04
            7.29008388828613629328856404754333e-01 1.43612885176155349008231265628410e-03 2.69555482319624839515626035790774e-01 6.22334338912881260968967822577724e-04
            6.40801775389629346690867350844201e-01 1.90358266676466281359936250083820e-03 3.57294641943605995049182411094080e-01 8.90476072747810234314647104270080e-04
            5.47506254918818435761806995287770e-01 2.39800530999366217840829129670510e-03 4.50095739771187886013592560630059e-01 1.16382522716152662749233837047314e-03
            4.52493745081181064637831923391786e-01 2.90152719418148219085451522403218e-03 5.44604727724637527330742159392685e-01 1.40819977829512296349512290305483e-03
            3.59198224610370542286830186640145e-01 3.39594983741047960409953354599111e-03 6.37405825552219029717093690123875e-01 1.58858983497967530540417868678560e-03
            2.70991611171386148626538670214359e-01 3.86340365241358936129745238474698e-03 7.25144985176200296272952527942834e-01 1.67417342464032045667998715998692e-03
            1.91061877798678114714903131243773e-01 4.28699387247230816883059034694270e-03 8.04651128328849551962775876745582e-01 1.64291248068185158880127882952138e-03
            1.22297795822498389650689887275803e-01 4.65141136002483659289685746784926e-03 8.73050792817476772889051517267944e-01 1.48506470566321744503079926857936e-03
            6.71843988060842889353807549923658e-02 4.94348659892883228461624156580001e-03 9.27872114594986863167491719650570e-01 1.20509870391844169795758645591377e-03
            2.77124884633839219105766460415907e-02 5.15266927079185976801056767726550e-03 9.67134842265824246076988401910057e-01 8.21746428634545652246179425759465e-04
            5.29953250417514176717759255552664e-03 5.27144745941233311414819695528422e-03 9.89429020036412532057568114396418e-01 3.66674469847455856627571613159944e-04
            ];
        
    case  30
        
        % ALG. DEG.:   30
        % PTS CARD.:  256
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.7e-15
        
        xyw_bar=[
            9.94700467495824969255124869960127e-01 5.27144745941222295920747242803373e-03 2.80850447628077856676576118388766e-05 1.95355620602029023035665737662026e-06
            9.72287511536616300134028278989717e-01 2.75656252300003658251004168278087e-02 1.46863233383334040871304182473978e-04 2.34217123568435754352168309821991e-05
            9.32815601193915933109224170038942e-01 6.68283529008377730207257627625950e-02 3.56045905246293870050067198462784e-04 8.67951091524684205467249364929216e-05
            8.77702204177501554838158881466370e-01 1.21649674678348099221558697990986e-01 6.48121144150345940282420542644104e-04 2.06926836109061045845466941806023e-04
            8.08938122201322107329701793787535e-01 1.90049339166975139736592836925411e-01 1.01253863170275293370536928705405e-03 3.88037026569801085824906738963591e-04
            7.29008388828613629328856404754333e-01 2.69555482319624839515626035790774e-01 1.43612885176153115551755945489276e-03 6.22334338912883429373312793586592e-04
            6.40801775389629346690867350844201e-01 3.57294641943606106071484873609734e-01 1.90358266676454723764777554606553e-03 8.90476072747813378500947312232938e-04
            5.47506254918818435761806995287770e-01 4.50095739771187941524743791887886e-01 2.39800530999362271344921282434370e-03 1.16382522716153053062015931828910e-03
            4.52493745081181064637831923391786e-01 5.44604727724637527330742159392685e-01 2.90152719418146354257714847335592e-03 1.40819977829512795082511633637523e-03
            3.59198224610370542286830186640145e-01 6.37405825552219029717093690123875e-01 3.39594983741042799607612323598005e-03 1.58858983497968094325547561140866e-03
            2.70991611171386148626538670214359e-01 7.25144985176200296272952527942834e-01 3.86340365241355510050880184280686e-03 1.67417342464032631137171858171087e-03
            1.91061877798678114714903131243773e-01 8.04651128328849662985078339261236e-01 4.28699387247222230001852949499153e-03 1.64291248068185722665257575414444e-03
            1.22297795822498389650689887275803e-01 8.73050792817476772889051517267944e-01 4.65141136002478194910736419842578e-03 1.48506470566322264920122719900064e-03
            6.71843988060842889353807549923658e-02 9.27872114594986974189794182166224e-01 4.94348659892873687482506284140982e-03 1.20509870391844581792584190083062e-03
            2.77124884633839219105766460415907e-02 9.67134842265824357099290864425711e-01 5.15266927079172099013248953269795e-03 8.21746428634548579592045136621437e-04
            5.29953250417514176717759255552664e-03 9.89429020036412643079870576912072e-01 5.27144745941221515295183053240180e-03 3.66674469847457103460069971490043e-04
            9.94700467495824969255124869960127e-01 5.15266927079175308251679510362919e-03 1.46863233383277662358334936243409e-04 4.47899603467711012274622695006343e-06
            9.72287511536616300134028278989717e-01 2.69445064465504702244658119525411e-02 7.67982016833229641505909057741519e-04 5.36998917401819832271453458716337e-05
            9.32815601193915933109224170038942e-01 6.53225519292510992119815682599437e-02 1.86184687683296767879426170111401e-03 1.98998599848442382729413191455592e-04
            8.77702204177501554838158881466370e-01 1.18908619566670148426368314176216e-01 3.38917625582829673547280435741413e-03 4.74429389614981127185761033970834e-04
            8.08938122201322107329701793787535e-01 1.85767077714389661213800764016923e-01 5.29480008428823145649744219554123e-03 8.89667928651336620271550259531068e-04
            7.29008388828613629328856404754333e-01 2.63481759273125559772665837954264e-01 7.50985189826081089847775729140267e-03 1.42685069804705387704057972797500e-03
            6.40801775389629346690867350844201e-01 3.49243947954787914422780659151613e-01 9.95427665558273888635199000418652e-03 2.04162991907839508690103436094887e-03
            5.47506254918818435761806995287770e-01 4.39954017390866036762986368557904e-01 1.25397276903155274752066361543257e-02 2.66834840044501923200792603552145e-03
            4.52493745081181064637831923391786e-01 5.32333494145750818482554223010084e-01 1.51727607730681723907650848559570e-02 3.22863565613302518075844460554435e-03
            3.59198224610370542286830186640145e-01 6.23043563581828552244701313611586e-01 1.77582118078009054684684997482691e-02 3.64222311581059937993232189512582e-03
            2.70991611171386148626538670214359e-01 7.08805752263490962405967366066761e-01 2.02026365651228889674939637188800e-02 3.83844401672051808299679365177326e-03
            1.91061877798678114714903131243773e-01 7.86520433822226472386773821199313e-01 2.24176883790954128983230475569144e-02 3.76677080680787139399523510974177e-03
            1.22297795822498389650689887275803e-01 8.53378891969946096196508733555675e-01 2.43233122075554586416501479106955e-02 3.40486693313773926966581129249789e-03
            6.71843988060842889353807549923658e-02 9.06964959607365006633017401327379e-01 2.58506415865507044316018436802551e-02 2.76297774264764716550812906348256e-03
            2.77124884633839219105766460415907e-02 9.45343005090065524598230695119128e-01 2.69445064465505534911926588392816e-02 1.88405072964969587563655029072152e-03
            5.29953250417514176717759255552664e-03 9.67134842265824468121593326941365e-01 2.75656252300003901112290805031080e-02 8.40689144956720062727695275128781e-04
            9.94700467495824969255124869960127e-01 4.94348659892872993593115893418144e-03 3.56045905246300808943971105691162e-04 6.84643325431189101208315217483857e-06
            9.72287511536616300134028278989717e-01 2.58506415865506732065792760977274e-02 1.86184687683302665939244491255522e-03 8.20837352202376843279124640773148e-05
            9.32815601193915933109224170038942e-01 6.26706553631491136702180710926768e-02 4.51374344293495322055775886838092e-03 3.04182147297234369009988341758799e-04
            8.77702204177501554838158881466370e-01 1.14081291934854611724858841625974e-01 8.21650388764383343698227690765634e-03 7.25195808331868222058880224523136e-04
            8.08938122201322107329701793787535e-01 1.78225500404012277000376229807443e-01 1.28363773946656156699219764050213e-02 1.35991459801602454851576773364741e-03
            7.29008388828613629328856404754333e-01 2.52785202693344701607713886915008e-01 1.82064084780416690634297083306592e-02 2.18103297980519863011328318691540e-03
            6.40801775389629346690867350844201e-01 3.35065707837710224925586999233929e-01 2.41325167726604283835456499218708e-02 3.12076252418117677800424303313775e-03
            5.47506254918818435761806995287770e-01 4.22093224854388937572480244853068e-01 3.04005202267926266657127598591615e-02 4.07874199518323734409142033996432e-03
            4.52493745081181064637831923391786e-01 5.10722376339527550648256237764144e-01 3.67838785792914402250630701018963e-02 4.93517706893878849855328283524614e-03
            3.59198224610370542286830186640145e-01 5.97749893356205874717090864578495e-01 4.30518820334235829960789487813599e-02 5.56737207772652325371565140699204e-03
            2.70991611171386148626538670214359e-01 6.80030398500571453546115208155243e-01 4.89779903280423978273461216303986e-02 5.86730833370437653007112288605640e-03
            1.91061877798678114714903131243773e-01 7.54590100789903517330969862086931e-01 5.43480214114183679541270066692960e-02 5.75775122671207378183266811788599e-03
            1.22297795822498389650689887275803e-01 8.18734309259061210362062865897315e-01 5.89678949184403444760960155690555e-02 5.20455790026651786417399847550769e-03
            6.71843988060842889353807549923658e-02 8.70144945830766625149976789543871e-01 6.26706553631490859146424554637633e-02 4.22338902551632790205449197173948e-03
            2.77124884633839219105766460415907e-02 9.06964959607365006633017401327379e-01 6.53225519292510714564059526310302e-02 2.87989260727580667978409145746355e-03
            5.29953250417514176717759255552664e-03 9.27872114594987085212096644681878e-01 6.68283529008377730207257627625950e-02 1.28504737981659038735526845442791e-03
            9.94700467495824969255124869960127e-01 4.65141136002473858102046477824842e-03 6.48121144150292163854665261624177e-04 8.96676417241144234750258729116368e-06
            9.72287511536616300134028278989717e-01 2.43233122075554065999458686064827e-02 3.38917625582829326602585240379995e-03 1.07504954590331375046362294423119e-04
            9.32815601193915933109224170038942e-01 5.89678949184402889649447843112284e-02 8.21650388764377792583104564982932e-03 3.98386937980330102255371871322609e-04
            8.77702204177501554838158881466370e-01 1.07341044959456871632497154678276e-01 1.49567508630415735293439638553536e-02 9.49787948058614258338427926986469e-04
            8.08938122201322107329701793787535e-01 1.67695431278192086388401094154688e-01 2.33664465204858062818971120577771e-02 1.78107826982021387289545355514520e-03
            7.29008388828613629328856404754333e-01 2.37849934438738280784164658143709e-01 3.31416767326480898869789371019579e-02 2.85649587978501449947454915445633e-03
            6.40801775389629346690867350844201e-01 3.15269073477167649510732871931395e-01 4.39291511332030037983997772244038e-02 4.08725836548662173342005843323932e-03
            5.47506254918818435761806995287770e-01 3.97154757434285576866983547006384e-01 5.53389876468959873712094577058451e-02 5.34192275487170315645402141058184e-03
            4.52493745081181064637831923391786e-01 4.80547446743216477571536415780429e-01 6.69588081756025133017828920856118e-02 6.46359458750240915836382527004389e-03
            3.59198224610370542286830186640145e-01 5.62433130700334071860879703308456e-01 7.83686446892953858522901100513991e-02 7.29157951691145837097574045060355e-03
            2.70991611171386148626538670214359e-01 6.39852269738763523854174763982883e-01 8.91561190898503275192865658027586e-02 7.68440561689799107869536243242692e-03
            1.91061877798678114714903131243773e-01 7.10006772899309357427455324796028e-01 9.89313493020125278576415439601988e-02 7.54091882526193232894051732273510e-03
            1.22297795822498389650689887275803e-01 7.70361159218044599938934879901353e-01 1.07341044959456954899224001565017e-01 6.81640229004776450338898285963296e-03
            6.71843988060842889353807549923658e-02 8.18734309259061099339760403381661e-01 1.14081291934854611724858841625974e-01 5.53136677061809214211995922028109e-03
            2.77124884633839219105766460415907e-02 8.53378891969945874151903808524366e-01 1.18908619566670203937519545434043e-01 3.77179136816235306695066498150481e-03
            5.29953250417514176717759255552664e-03 8.73050792817476772889051517267944e-01 1.21649674678348085343770890176529e-01 1.68302477759986663687075125750425e-03
            9.94700467495824969255124869960127e-01 4.28699387247221969793331552978088e-03 1.01253863170281104694181451009172e-03 1.07630829280184121989191389778817e-05
            9.72287511536616300134028278989717e-01 2.24176883790953677955126721599299e-02 5.29480008428833207045904885035270e-03 1.29041504736865868903025744707236e-04
            9.32815601193915933109224170038942e-01 5.43480214114183887708087183909811e-02 1.28363773946656781199671115700767e-02 4.78196099336967214016302918366819e-04
            8.77702204177501554838158881466370e-01 9.89313493020125417354293517746555e-02 2.33664465204859034264117667589744e-02 1.14005969739227794833291707021772e-03
            8.08938122201322107329701793787535e-01 1.54557236650721019355358976099524e-01 3.65046411479568733149392301129410e-02 2.13788304797260346468146252618681e-03
            7.29008388828613629328856404754333e-01 2.19215445073292103383977291741758e-01 5.17761660980942672871663035039091e-02 3.42873989395896120249696004123052e-03
            6.40801775389629346690867350844201e-01 2.90569137314362013757573777184007e-01 6.86290872960086395515588719717925e-02 4.90606197398618659732960622932296e-03
            5.47506254918818435761806995287770e-01 3.66039440453814757248096611874644e-01 8.64543046273668069900963928375859e-02 6.41207424442504989037239226945530e-03
            4.52493745081181064637831923391786e-01 4.42898681747507794170815031975508e-01 1.04607573171311196702504275890533e-01 7.75845145703997398495266324403019e-03
            3.59198224610370542286830186640145e-01 5.18368984886960260105581710377010e-01 1.22432790502669197607588102982845e-01 8.75230724347830779663581779459491e-03
            2.70991611171386148626538670214359e-01 5.89722677128030170479178195819259e-01 1.39285711700583680894283133966383e-01 9.22382849787392426754717433823316e-03
            1.91061877798678114714903131243773e-01 6.54380885550600921440889123914531e-01 1.54557236650720963844207744841697e-01 9.05159688703198635206792488361316e-03
            1.22297795822498389650689887275803e-01 7.10006772899309468449757787311682e-01 1.67695431278192086388401094154688e-01 8.18193739768985350835350800480228e-03
            6.71843988060842889353807549923658e-02 7.54590100789903517330969862086931e-01 1.78225500404012193733649382920703e-01 6.63946972539116320871643850409782e-03
            2.77124884633839219105766460415907e-02 7.86520433822226472386773821199313e-01 1.85767077714389605702649532759096e-01 4.52739723072229181127790198502225e-03
            5.29953250417514176717759255552664e-03 8.04651128328849774007380801776890e-01 1.90049339166975084225441605667584e-01 2.02018642432363025596142591666649e-03
            9.94700467495824969255124869960127e-01 3.86340365241350739561321248061176e-03 1.43612885176152334926191755926084e-03 1.21704175390405777148431637479931e-05
            9.72287511536616300134028278989717e-01 2.02026365651228542730244441827381e-02 7.50985189826084559294727682754456e-03 1.45914419039311658377391922591926e-04
            9.32815601193915933109224170038942e-01 4.89779903280423770106644099087134e-02 1.82064084780416898801114200523443e-02 5.40722972534311428945141830837429e-04
            8.77702204177501554838158881466370e-01 8.91561190898502997637109501738450e-02 3.31416767326481453981301683597849e-02 1.28912901902640975732028394418194e-03
            8.08938122201322107329701793787535e-01 1.39285711700583680894283133966383e-01 5.17761660980942117760150722460821e-02 2.41742347591979763604674857901955e-03
            7.29008388828613629328856404754333e-01 1.97555157846122514397890768123034e-01 7.34364533252638562732528271226329e-02 3.87706723262505039903635939424476e-03
            6.40801775389629346690867350844201e-01 2.61858518993304811850464375311276e-01 9.73397056170658414586682738445234e-02 5.54755761849495473980731219398876e-03
            5.47506254918818435761806995287770e-01 3.29871736056657594016172652118257e-01 1.22622009024523970222020352593972e-01 7.25048960115635531681332892617320e-03
            4.52493745081181064637831923391786e-01 3.99136652771956423890742371440865e-01 1.48369602146862566982576936425176e-01 8.77291332976271390209355161005078e-03
            3.59198224610370542286830186640145e-01 4.67149869835308928500694491958711e-01 1.73651905554320529212475321401143e-01 9.89672144082527453778919124260938e-03
            2.70991611171386148626538670214359e-01 5.31453230982491309219994946033694e-01 1.97555157846122542153466383751947e-01 1.04298967942909708017662495649347e-02
            1.91061877798678114714903131243773e-01 5.89722677128029837412270808272297e-01 2.19215445073292047872826060483931e-01 1.02351449159131420441060100756658e-02
            1.22297795822498389650689887275803e-01 6.39852269738763301809569838951575e-01 2.37849934438738253028589042514795e-01 9.25177247765662444378698126001836e-03
            6.71843988060842889353807549923658e-02 6.80030398500571120479207820608281e-01 2.52785202693344590585411424399354e-01 7.50761834097533539700286553397746e-03
            2.77124884633839219105766460415907e-02 7.08805752263490518316757516004145e-01 2.63481759273125559772665837954264e-01 5.11938029572822639323703697300516e-03
            5.29953250417514176717759255552664e-03 7.25144985176200074228347602911526e-01 2.69555482319624784004474804532947e-01 2.28433734601416827195152770002551e-03
            9.94700467495824969255124869960127e-01 3.39594983741040804675614950269846e-03 1.90358266676462269811898053717414e-03 1.31378903579940360359578840898642e-05
            9.72287511536616300134028278989717e-01 1.77582118078008603656581243512846e-02 9.95427665558283950031359665899799e-03 1.57513711656929398510426643120752e-04
            9.32815601193915933109224170038942e-01 4.30518820334235621793972370596748e-02 2.41325167726605047113785929013829e-02 5.83707099975505562701461048646934e-04
            8.77702204177501554838158881466370e-01 7.83686446892953580967144944224856e-02 4.39291511332030870651266241111443e-02 1.39160679203883066480662566988258e-03
            8.08938122201322107329701793787535e-01 1.22432790502669239240951526426215e-01 6.86290872960086534293466797862493e-02 2.60959367035642694765407156864967e-03
            7.29008388828613629328856404754333e-01 1.73651905554320501456899705772230e-01 9.73397056170658692142438894734369e-02 4.18527006566569385187515095481103e-03
            6.40801775389629346690867350844201e-01 2.30174860047128393425452941301046e-01 1.29023364563242259883679707854753e-01 5.98855409131565218777693004881257e-03
            5.47506254918818435761806995287770e-01 2.89958795200723484786209382946254e-01 1.62534949880458079451983621765976e-01 7.82685861978054710219598888443215e-03
            4.52493745081181064637831923391786e-01 3.50842980188906194971565355444909e-01 1.96663274729912795901753952421132e-01 9.47030560594003430197052040284689e-03
            3.59198224610370542286830186640145e-01 4.10626915342501064287716872058809e-01 2.30174860047128393425452941301046e-01 1.06834495017186694582056460944841e-02
            2.70991611171386148626538670214359e-01 4.67149869835308984011845723216538e-01 2.61858518993304867361615606569103e-01 1.12590089936544929249961910500133e-02
            1.91061877798678114714903131243773e-01 5.18368984886960038060976785345702e-01 2.90569137314361847224120083410526e-01 1.10487755471080899977165401537604e-02
            1.22297795822498389650689887275803e-01 5.62433130700333960838577240792802e-01 3.15269073477167593999581640673568e-01 9.98723109035925052379134569946473e-03
            6.71843988060842889353807549923658e-02 5.97749893356205652672485939547187e-01 3.35065707837710058392133305460447e-01 8.10442750193224661914292994424613e-03
            2.77124884633839219105766460415907e-02 6.23043563581828330200096388580278e-01 3.49243947954787747889326965378132e-01 5.52633932323201310793381324515394e-03
            5.29953250417514176717759255552664e-03 6.37405825552218918694791227608221e-01 3.57294641943605939538031179836253e-01 2.46592801736949217283889623786308e-03
            9.94700467495824969255124869960127e-01 2.90152719418141843976677307637146e-03 2.39800530999361230510835696350114e-03 1.36305301179685297289405701559595e-05
            9.72287511536616300134028278989717e-01 1.51727607730681411657425172734293e-02 1.25397276903155587002292037368534e-02 1.63420102636676424284409292475573e-04
            9.32815601193915933109224170038942e-01 3.67838785792914263472752622874395e-02 3.04005202267926405435005676736182e-02 6.05594733209738680934641763542459e-04
            8.77702204177501554838158881466370e-01 6.69588081756024161572682373844145e-02 5.53389876468960290045728811492154e-02 1.44378874951663397317724868429423e-03
            8.08938122201322107329701793787535e-01 1.04607573171311099557989621189336e-01 8.64543046273667931123085850231291e-02 2.70744724991633760324827306931184e-03
            7.29008388828613629328856404754333e-01 1.48369602146862400449123242651694e-01 1.22622009024523970222020352593972e-01 4.34220777669807548454494749989863e-03
            6.40801775389629346690867350844201e-01 1.96663274729912629368300258647650e-01 1.62534949880458023940832390508149e-01 6.21311068067285487276363298292381e-03
            5.47506254918818435761806995287770e-01 2.47743155743588239214858504055883e-01 2.04750589337593325023334500656347e-01 8.12034727334179709523631629508600e-03
            4.52493745081181064637831923391786e-01 2.99763099175230474102704647521023e-01 2.47743155743588516770614660345018e-01 9.82541962755741071777748629756388e-03
            3.59198224610370542286830186640145e-01 3.50842980188905917415809199155774e-01 2.89958795200723540297360614204081e-01 1.10840535450477326029350422231801e-02
            2.70991611171386148626538670214359e-01 3.99136652771956146334986215151730e-01 3.29871736056657705038475114633911e-01 1.16811951542209538357219855697622e-02
            1.91061877798678114714903131243773e-01 4.42898681747507239059302719397238e-01 3.66039440453814646225794149358990e-01 1.14630784515487131602906600846836e-02
            1.22297795822498389650689887275803e-01 4.80547446743215977971175334459986e-01 3.97154757434285576866983547006384e-01 1.03617285928574824616088534412484e-02
            6.71843988060842889353807549923658e-02 5.10722376339526884514441462670220e-01 4.22093224854388826550177782337414e-01 8.40832433091234614230824462310920e-03
            2.77124884633839219105766460415907e-02 5.32333494145750152348739447916159e-01 4.39954017390865925740683906042250e-01 5.73356395394131624493327947789112e-03
            5.29953250417514176717759255552664e-03 5.44604727724636972219229846814414e-01 4.50095739771187886013592560630059e-01 2.55839447533868233689013926834832e-03
            9.94700467495824969255124869960127e-01 2.39800530999360970302314299829050e-03 2.90152719418142104185198704158211e-03 1.36305301179684992357544690011473e-05
            9.72287511536616300134028278989717e-01 1.25397276903154893112901646645696e-02 1.51727607730682105546815563457130e-02 1.63420102636676071918703234686632e-04
            9.32815601193915933109224170038942e-01 3.04005202267926093184780000910905e-02 3.67838785792914541028508779163531e-02 6.05594733209737271471817532386694e-04
            8.77702204177501554838158881466370e-01 5.53389876468959179822704186335613e-02 6.69588081756025271795706999000686e-02 1.44378874951663072057073122778093e-03
            8.08938122201322107329701793787535e-01 8.64543046273667514789451615797589e-02 1.04607573171311141191353044632706e-01 2.70744724991633153171610715048701e-03
            7.29008388828613629328856404754333e-01 1.22622009024523845321930082263862e-01 1.48369602146862511471425705167348e-01 4.34220777669806594356582962745961e-03
            6.40801775389629346690867350844201e-01 1.62534949880457912918529927992495e-01 1.96663274729912740390602721163305e-01 6.21311068067284099497582516846705e-03
            5.47506254918818435761806995287770e-01 2.04750589337593102978729575625039e-01 2.47743155743588461259463429087191e-01 8.12034727334177801327808055020796e-03
            4.52493745081181064637831923391786e-01 2.47743155743588211459282888426969e-01 2.99763099175230807169612035067985e-01 9.82541962755738816637229859907166e-03
            3.59198224610370542286830186640145e-01 2.89958795200723262741604457914946e-01 3.50842980188906194971565355444909e-01 1.10840535450477083168063785478807e-02
            2.70991611171386148626538670214359e-01 3.29871736056657316460416495829122e-01 3.99136652771956534913044833956519e-01 1.16811951542209278148698459176558e-02
            1.91061877798678114714903131243773e-01 3.66039440453814257647735530554201e-01 4.42898681747507627637361338202027e-01 1.14630784515486871394385204325772e-02
            1.22297795822498389650689887275803e-01 3.97154757434285132777773696943768e-01 4.80547446743216422060385184522602e-01 1.03617285928574599102036657427561e-02
            6.71843988060842889353807549923658e-02 4.22093224854388382460967932274798e-01 5.10722376339527328603651312732836e-01 8.40832433091232706035000887823116e-03
            2.77124884633839219105766460415907e-02 4.39954017390865426140322824721807e-01 5.32333494145750707460251760494430e-01 5.73356395394130323450720965183791e-03
            5.29953250417514176717759255552664e-03 4.50095739771187386413231479309616e-01 5.44604727724637527330742159392685e-01 2.55839447533867713271971133792704e-03
            9.94700467495824969255124869960127e-01 1.90358266676462226443811154297236e-03 3.39594983741040848043701849690024e-03 1.31378903579940258715625170382602e-05
            9.72287511536616300134028278989717e-01 9.95427665558278051971541344755678e-03 1.77582118078009193462563075627259e-02 1.57513711656929290090209394570309e-04
            9.32815601193915933109224170038942e-01 2.41325167726604943030377370405404e-02 4.30518820334235691182911409669032e-02 5.83707099975505020600374805894717e-04
            8.77702204177501554838158881466370e-01 4.39291511332030107372936811316322e-02 7.83686446892954413634413413092261e-02 1.39160679203882958060445318437814e-03
            8.08938122201322107329701793787535e-01 6.86290872960086950627101032296196e-02 1.22432790502669197607588102982845e-01 2.60959367035642477924972659764080e-03
            7.29008388828613629328856404754333e-01 9.73397056170658414586682738445234e-02 1.73651905554320529212475321401143e-01 4.18527006566569038242819900119684e-03
            6.40801775389629346690867350844201e-01 1.29023364563242259883679707854753e-01 2.30174860047128393425452941301046e-01 5.98855409131564785096824010679484e-03
            5.47506254918818435761806995287770e-01 1.62534949880457996185256774879235e-01 2.89958795200723540297360614204081e-01 7.82685861978054016330208497720378e-03
            4.52493745081181064637831923391786e-01 1.96663274729912740390602721163305e-01 3.50842980188906250482716586702736e-01 9.47030560594002736307661649561851e-03
            3.59198224610370542286830186640145e-01 2.30174860047128337914301710043219e-01 4.10626915342501119798868103316636e-01 1.06834495017186625193117421872557e-02
            2.70991611171386148626538670214359e-01 2.61858518993304756339313144053449e-01 4.67149869835309095034148185732192e-01 1.12590089936544842513788111659778e-02
            1.91061877798678114714903131243773e-01 2.90569137314361791712968852152699e-01 5.18368984886960149083279247861356e-01 1.10487755471080813240991602697250e-02
            1.22297795822498389650689887275803e-01 3.15269073477167482977279178157914e-01 5.62433130700334071860879703308456e-01 9.98723109035924185017396581542926e-03
            6.71843988060842889353807549923658e-02 3.35065707837709947369830842944793e-01 5.97749893356205763694788402062841e-01 8.10442750193224141497250201382485e-03
            2.77124884633839219105766460415907e-02 3.49243947954787636867024502862478e-01 6.23043563581828441222398851095932e-01 5.52633932323200877112512330313621e-03
            5.29953250417514176717759255552664e-03 3.57294641943605884026879948578426e-01 6.37405825552219029717093690123875e-01 2.46592801736949043811542026105599e-03
            9.94700467495824969255124869960127e-01 1.43612885176152226505974507375640e-03 3.86340365241350826297495046901531e-03 1.21704175390405218106686449641707e-05
            9.72287511536616300134028278989717e-01 7.50985189826078487762561763929625e-03 2.02026365651229167230695793477935e-02 1.45914419039310980751034119151655e-04
            9.32815601193915933109224170038942e-01 1.82064084780416725328766602842734e-02 4.89779903280423978273461216303986e-02 5.40722972534308935280145114177230e-04
            8.77702204177501554838158881466370e-01 3.31416767326480551925094175658160e-02 8.91561190898503830304377970605856e-02 1.28912901902640390262855252245799e-03
            8.08938122201322107329701793787535e-01 5.17761660980942325926967839677673e-02 1.39285711700583653138707518337469e-01 2.41742347591978636034415472977344e-03
            7.29008388828613629328856404754333e-01 7.34364533252638007621015958648059e-02 1.97555157846122569909041999380861e-01 3.87706723262503261812073063197204e-03
            6.40801775389629346690867350844201e-01 9.73397056170657998253048504011531e-02 2.61858518993304867361615606569103e-01 5.54755761849492958631691053028590e-03
            5.47506254918818435761806995287770e-01 1.22622009024523873077505697892775e-01 3.29871736056657705038475114633911e-01 7.25048960115632148970554737843486e-03
            4.52493745081181064637831923391786e-01 1.48369602146862428204698858280608e-01 3.99136652771956534913044833956519e-01 8.77291332976267400345360414348761e-03
            3.59198224610370542286830186640145e-01 1.73651905554320390434597243256576e-01 4.67149869835309039522996954474365e-01 9.89672144082522943497881584562492e-03
            2.70991611171386148626538670214359e-01 1.97555157846122403375588305607380e-01 5.31453230982491420242297408549348e-01 1.04298967942909222295089222143361e-02
            1.91061877798678114714903131243773e-01 2.19215445073291881339372366710450e-01 5.89722677128029948434573270787951e-01 1.02351449159130952065721587018743e-02
            1.22297795822498389650689887275803e-01 2.37849934438738086495135348741314e-01 6.39852269738763412831872301467229e-01 9.25177247765658281042355781664810e-03
            6.71843988060842889353807549923658e-02 2.52785202693344424051957730625873e-01 6.80030398500571342523812745639589e-01 7.50761834097530070253334599783557e-03
            2.77124884633839219105766460415907e-02 2.63481759273125282216909681665129e-01 7.08805752263490740361362441035453e-01 5.11938029572820297447011128610939e-03
            5.29953250417514176717759255552664e-03 2.69555482319624617471021110759466e-01 7.25144985176200185250650065427180e-01 2.28433734601415786361067183918294e-03
            9.94700467495824969255124869960127e-01 1.01253863170281234798442149269704e-03 4.28699387247221883057157754137734e-03 1.07630829280184291395780840638885e-05
            9.72287511536616300134028278989717e-01 5.29480008428828263083998351135051e-03 2.24176883790954163677699995105286e-02 1.29041504736866085743460241808123e-04
            9.32815601193915933109224170038942e-01 1.28363773946656902630314434077263e-02 5.43480214114183748930209105765243e-02 4.78196099336968027167932282495144e-04
            8.77702204177501554838158881466370e-01 2.33664465204858756708361511300609e-02 9.89313493020125694910049674035690e-02 1.14005969739227989989682754412570e-03
            8.08938122201322107329701793787535e-01 3.65046411479569773983477887213667e-02 1.54557236650720908333056513583870e-01 2.13788304797260693412841447980099e-03
            7.29008388828613629328856404754333e-01 5.17761660980943158594236308545078e-02 2.19215445073292047872826060483931e-01 3.42873989395896684034825696585358e-03
            6.40801775389629346690867350844201e-01 6.86290872960087783294369501163601e-02 2.90569137314361847224120083410526e-01 4.90606197398619527094698611335843e-03
            5.47506254918818435761806995287770e-01 8.64543046273669180123988553532399e-02 3.66039440453814646225794149358990e-01 6.41207424442506029871324813029787e-03
            4.52493745081181064637831923391786e-01 1.04607573171311307724806738406187e-01 4.42898681747507683148512569459854e-01 7.75845145703998699537873307008340e-03
            3.59198224610370542286830186640145e-01 1.22432790502669364141041796756326e-01 5.18368984886960149083279247861356e-01 8.75230724347832340914710158585876e-03
            2.70991611171386148626538670214359e-01 1.39285711700583847427736827739864e-01 5.89722677128029948434573270787951e-01 9.22382849787393988005845812949701e-03
            1.91061877798678114714903131243773e-01 1.54557236650721102622085822986264e-01 6.54380885550600810418586661398876e-01 9.05159688703200196457920867487701e-03
            1.22297795822498389650689887275803e-01 1.67695431278192225166279172299255e-01 7.10006772899309357427455324796028e-01 8.18193739768986738614131581925903e-03
            6.71843988060842889353807549923658e-02 1.78225500404012388022678692323097e-01 7.54590100789903295286364937055623e-01 6.63946972539117448441903235334394e-03
            2.77124884633839219105766460415907e-02 1.85767077714389772236103226532578e-01 7.86520433822226250342168896168005e-01 4.52739723072229961753354388065418e-03
            5.29953250417514176717759255552664e-03 1.90049339166975278514470915069978e-01 8.04651128328849551962775876745582e-01 2.02018642432363372540837787028067e-03
            9.94700467495824969255124869960127e-01 6.48121144150291513333361770321517e-04 4.65141136002473944838220276665197e-03 8.96676417241144065343669278256300e-06
            9.72287511536616300134028278989717e-01 3.38917625582823515278940718076228e-03 2.43233122075554655805440518179239e-02 1.07504954590331361493835138354314e-04
            9.32815601193915933109224170038942e-01 8.21650388764377445638409369621513e-03 5.89678949184402889649447843112284e-02 3.98386937980330048045263247047387e-04
            8.77702204177501554838158881466370e-01 1.49567508630415041404049247830699e-02 1.07341044959456941021436193750560e-01 9.49787948058614041497993429885582e-04
            8.08938122201322107329701793787535e-01 2.33664465204858548541544394083758e-02 1.67695431278192030877249862896861e-01 1.78107826982021343921458456094342e-03
            7.29008388828613629328856404754333e-01 3.31416767326480829480850331947295e-02 2.37849934438738280784164658143709e-01 2.85649587978501363211281116605278e-03
            6.40801775389629346690867350844201e-01 4.39291511332030315539753928533173e-02 3.15269073477167593999581640673568e-01 4.08725836548662086605832044483577e-03
            5.47506254918818435761806995287770e-01 5.53389876468959804323155537986167e-02 3.97154757434285576866983547006384e-01 5.34192275487170142173054543377475e-03
            4.52493745081181064637831923391786e-01 6.69588081756024855462072764566983e-02 4.80547446743216477571536415780429e-01 6.46359458750240829100208728164034e-03
            3.59198224610370542286830186640145e-01 7.83686446892953719745023022369423e-02 5.62433130700334071860879703308456e-01 7.29157951691145663625226447379646e-03
            2.70991611171386148626538670214359e-01 8.91561190898503275192865658027586e-02 6.39852269738763523854174763982883e-01 7.68440561689798934397188645561982e-03
            1.91061877798678114714903131243773e-01 9.89313493020125139798537361457420e-02 7.10006772899309357427455324796028e-01 7.54091882526193059421704134592801e-03
            1.22297795822498389650689887275803e-01 1.07341044959456871632497154678276e-01 7.70361159218044710961237342417007e-01 6.81640229004776276866550688282587e-03
            6.71843988060842889353807549923658e-02 1.14081291934854583969283225997060e-01 8.18734309259061099339760403381661e-01 5.53136677061809127475822123187754e-03
            2.77124884633839219105766460415907e-02 1.18908619566670120670792698547302e-01 8.53378891969945985174206271040021e-01 3.77179136816235219958892699310127e-03
            5.29953250417514176717759255552664e-03 1.21649674678348085343770890176529e-01 8.73050792817476772889051517267944e-01 1.68302477759986620318988226330248e-03
            9.94700467495824969255124869960127e-01 3.56045905246301838936034966920374e-04 4.94348659892872906856942094577789e-03 6.84643325431193505779640939845621e-06
            9.72287511536616300134028278989717e-01 1.86184687683297678609251057935126e-03 2.58506415865507217788366034483261e-02 8.20837352202382128764715507607264e-05
            9.32815601193915933109224170038942e-01 4.51374344293496536362209070603058e-03 6.26706553631490997924302632782201e-02 3.04182147297236320573898815666780e-04
            8.77702204177501554838158881466370e-01 8.21650388764380047723623334832155e-03 1.14081291934854639480434457254887e-01 7.25195808331872884128221912192203e-04
            8.08938122201322107329701793787535e-01 1.28363773946657214880540109902540e-02 1.78225500404012165978073767291789e-01 1.35991459801603322213314761768288e-03
            7.29008388828613629328856404754333e-01 1.82064084780417488607096032637855e-02 2.52785202693344646096562655657181e-01 2.18103297980521250790109100137215e-03
            6.40801775389629346690867350844201e-01 2.41325167726605845086584878345093e-02 3.35065707837710058392133305460447e-01 3.12076252418119672732421676641934e-03
            5.47506254918818435761806995287770e-01 3.04005202267927446269091262820439e-02 4.22093224854388826550177782337414e-01 4.07874199518326336494355999207073e-03
            4.52493745081181064637831923391786e-01 3.67838785792915859418350521536922e-02 5.10722376339527439625953775248490e-01 4.93517706893882059093758840617738e-03
            3.59198224610370542286830186640145e-01 4.30518820334237148350631230186991e-02 5.97749893356205763694788402062841e-01 5.56737207772655881554690893153747e-03
            2.70991611171386148626538670214359e-01 4.89779903280425574219059114966512e-02 6.80030398500571342523812745639589e-01 5.86730833370441382662585638740893e-03
            1.91061877798678114714903131243773e-01 5.43480214114185553042624121644621e-02 7.54590100789903295286364937055623e-01 5.75775122671211021102566363083497e-03
            1.22297795822498389650689887275803e-01 5.89678949184404832539740937136230e-02 8.18734309259061099339760403381661e-01 5.20455790026655169128178002324603e-03
            6.71843988060842889353807549923658e-02 6.26706553631493079592473804950714e-02 8.70144945830766403105371864512563e-01 4.22338902551635479026836961224944e-03
            2.77124884633839219105766460415907e-02 6.53225519292512935010108776623383e-02 9.06964959607364784588412476296071e-01 2.87989260727582489438058921393804e-03
            5.29953250417514176717759255552664e-03 6.68283529008379811875428799794463e-02 9.27872114594986863167491719650570e-01 1.28504737981659862729177934426161e-03
            9.94700467495824969255124869960127e-01 1.46863233383278638140290173197400e-04 5.15266927079175221515505711522565e-03 4.47899603467706692406591698074614e-06
            9.72287511536616300134028278989717e-01 7.67982016833179226104888481785338e-04 2.69445064465505222661700912567539e-02 5.36998917401814750073769932914303e-05
            9.32815601193915933109224170038942e-01 1.86184687683298914599727691410180e-03 6.53225519292510714564059526310302e-02 1.98998599848440485375611341822832e-04
            8.77702204177501554838158881466370e-01 3.38917625582826897989718872850062e-03 1.18908619566670176181943929805129e-01 4.74429389614976627746745219127433e-04
            8.08938122201322107329701793787535e-01 5.29480008428833120309731086194915e-03 1.85767077714389550191498301501269e-01 8.89667928651328163494604872596483e-04
            7.29008388828613629328856404754333e-01 7.50985189826086640962898854922969e-03 2.63481759273125504261514606696437e-01 1.42685069804704043293364090772002e-03
            6.40801775389629346690867350844201e-01 9.95427665558288460312397205598245e-03 3.49243947954787747889326965378132e-01 2.04162991907837557126192962186906e-03
            5.47506254918818435761806995287770e-01 1.25397276903156298238917187859442e-02 4.39954017390865925740683906042250e-01 2.66834840044499407851752437181858e-03
            4.52493745081181064637831923391786e-01 1.51727607730683076991962110469103e-02 5.32333494145750707460251760494430e-01 3.22863565613299438941674601721843e-03
            3.59198224610370542286830186640145e-01 1.77582118078010407768996259392225e-02 6.23043563581828441222398851095932e-01 3.64222311581056468546280235898394e-03
            2.70991611171386148626538670214359e-01 2.02026365651230624398415613995894e-02 7.08805752263490740361362441035453e-01 3.83844401672048165380379813882428e-03
            1.91061877798678114714903131243773e-01 2.24176883790955863706706452376238e-02 7.86520433822226250342168896168005e-01 3.76677080680783539848310859099456e-03
            1.22297795822498389650689887275803e-01 2.43233122075556529306794573130901e-02 8.53378891969945874151903808524366e-01 3.40486693313770674360063672736487e-03
            6.71843988060842889353807549923658e-02 2.58506415865509264762067687115632e-02 9.06964959607364784588412476296071e-01 2.76297774264762114465598941137614e-03
            2.77124884633839219105766460415907e-02 2.69445064465507339024341604272195e-02 9.45343005090065302553625770087820e-01 1.88405072964967809472092152844880e-03
            5.29953250417514176717759255552664e-03 2.75656252300006399114096211633296e-02 9.67134842265824246076988401910057e-01 8.40689144956712039631618882395969e-04
            9.94700467495824969255124869960127e-01 2.80850447628082600061080742470665e-05 5.27144745941222209184573443963018e-03 1.95355620602028345409307934221754e-06
            9.72287511536616300134028278989717e-01 1.46863233383280264443548901454051e-04 2.75656252300004178668046961320215e-02 2.34217123568434941200538945693665e-05
            9.32815601193915933109224170038942e-01 3.56045905246308073098526758570870e-04 6.68283529008377591429379549481382e-02 8.67951091524681223911275029792023e-05
            8.77702204177501554838158881466370e-01 6.48121144150305065860517839126942e-04 1.21649674678348140854922121434356e-01 2.06926836109060314009000514090530e-04
            8.08938122201322107329701793787535e-01 1.01253863170283273098526422018040e-03 1.90049339166975056469865990038670e-01 3.88037026569799784782299756358270e-04
            7.29008388828613629328856404754333e-01 1.43612885176155349008231265628410e-03 2.69555482319624839515626035790774e-01 6.22334338912881260968967822577724e-04
            6.40801775389629346690867350844201e-01 1.90358266676466281359936250083820e-03 3.57294641943605995049182411094080e-01 8.90476072747810234314647104270080e-04
            5.47506254918818435761806995287770e-01 2.39800530999366217840829129670510e-03 4.50095739771187886013592560630059e-01 1.16382522716152662749233837047314e-03
            4.52493745081181064637831923391786e-01 2.90152719418148219085451522403218e-03 5.44604727724637527330742159392685e-01 1.40819977829512296349512290305483e-03
            3.59198224610370542286830186640145e-01 3.39594983741047960409953354599111e-03 6.37405825552219029717093690123875e-01 1.58858983497967530540417868678560e-03
            2.70991611171386148626538670214359e-01 3.86340365241358936129745238474698e-03 7.25144985176200296272952527942834e-01 1.67417342464032045667998715998692e-03
            1.91061877798678114714903131243773e-01 4.28699387247230816883059034694270e-03 8.04651128328849551962775876745582e-01 1.64291248068185158880127882952138e-03
            1.22297795822498389650689887275803e-01 4.65141136002483659289685746784926e-03 8.73050792817476772889051517267944e-01 1.48506470566321744503079926857936e-03
            6.71843988060842889353807549923658e-02 4.94348659892883228461624156580001e-03 9.27872114594986863167491719650570e-01 1.20509870391844169795758645591377e-03
            2.77124884633839219105766460415907e-02 5.15266927079185976801056767726550e-03 9.67134842265824246076988401910057e-01 8.21746428634545652246179425759465e-04
            5.29953250417514176717759255552664e-03 5.27144745941233311414819695528422e-03 9.89429020036412532057568114396418e-01 3.66674469847455856627571613159944e-04
            ];
        
    case  31
        
        % ALG. DEG.:   31
        % PTS CARD.:  289
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.1e-15
        
        xyw_bar=[
            9.95287737657208571029343602276640e-01 4.69005692640413890392991902444919e-03 2.22054163872900667264786989107961e-05 1.37395558340911968700894310591032e-06
            9.75337760884383841997191666450817e-01 2.45460241749427786339676771376617e-02 1.16214940673379368840656411521195e-04 1.65144951780179013035420448884949e-05
            9.40119576863492900642427230195608e-01 5.95982508734905214176080789911794e-02 2.82172263016577939964690813212655e-04 6.14815856611745350981251823085927e-05
            8.90757001948400684021578399551800e-01 1.08728216385667125765124296776776e-01 5.14781665932190213297303671424743e-04 1.47576640711611462930888394318174e-04
            8.28835579608345418556325512327021e-01 1.70357848739017270212059429468354e-01 8.06571652637311231615058204624802e-04 2.79281620774757049704972722281582e-04
            7.56345268543238580249976621416863e-01 2.42506566441074766782293181677232e-01 1.14816501568665296773019690590445e-03 4.53190940909956735405117456139124e-04
            6.75615881726938205531496350886300e-01 3.22855535207924038143545431012171e-01 1.52858306513775632495821810152847e-03 6.58015445203851941438633055270202e-04
            5.89242090747923774607386349089211e-01 4.08822310224303875347828807207406e-01 1.93559902777235004478484370338265e-03 8.75672130250856948878446228690109e-04
            5.00000000000000000000000000000000e-01 4.97643868828604285514671801138320e-01 2.35613117139571448532819886167999e-03 1.08333192871339144096320961097035e-03
            4.10757909252076280903764882168616e-01 5.86465427432904529148061101295752e-01 2.77666331501913443702278527780436e-03 1.25617271199530413858247435143767e-03
            3.24384118273061905490806111629354e-01 6.72432202449284366352344477490988e-01 3.18367927765372815684941087965854e-03 1.37049152581235475154008973674991e-03
            2.43654731456761419750023378583137e-01 7.52781171216133859758201651857235e-01 3.56409732710472049177496955962852e-03 1.40678090613944001101287550881125e-03
            1.71164420391654581443674487672979e-01 8.24929888918191300817284172808286e-01 3.90569069015411773904133951873519e-03 1.35237535639205655284278417838095e-03
            1.09242998051599260467270369190373e-01 8.86559521271541584042097383644432e-01 4.19748067685921100178347842302173e-03 1.20332587335071693017407579162636e-03
            5.98804231365069883352703072887380e-02 9.35689486783718216145189217058942e-01 4.43009007977479551954047565232031e-03 9.65257746507828823939734164127913e-04
            2.46622391156161580028083335491829e-02 9.70741713482265899948231435701018e-01 4.59604740211794204896023074979894e-03 6.53112260957066466035558782721182e-04
            4.71226234279131794835393520770594e-03 9.90597680730804563964397857489530e-01 4.69005692640411808724820730276406e-03 2.90196310132152649786940745713082e-04
            9.95287737657208571029343602276640e-01 4.59604740211799322330277206560822e-03 1.16214940673435747353625657751763e-04 3.15545694666094234939399121309123e-06
            9.75337760884383841997191666450817e-01 2.40540130774203851637960127618499e-02 6.08226038195772839012320787333010e-04 3.79275568725268691881724603121029e-05
            9.40119576863492900642427230195608e-01 5.84036378227702837606649666213343e-02 1.47678531373681559690780318305769e-03 1.41199976847079247069954854865159e-04
            8.90757001948400684021578399551800e-01 1.06548821111943983752645692675287e-01 2.69417693965533222577590777291334e-03 3.38927794843917019277507174379593e-04
            8.28835579608345418556325512327021e-01 1.66943122527869747262485589089920e-01 4.22129786378483418118889858305920e-03 6.41404380891127381611460656074541e-04
            7.56345268543238580249976621416863e-01 2.37645660207923520301775965890556e-01 6.00907124883789944824741269258084e-03 1.04080839288115214492924653910677e-03
            6.75615881726938205531496350886300e-01 3.16384079582903210425826046048314e-01 8.00003869015858404267760306538548e-03 1.51121290429693621963636029903455e-03
            5.89242090747923774607386349089211e-01 4.00627699475470888668127145137987e-01 1.01302097766053367244865057728020e-02 2.01108808739028669873571963933045e-03
            5.00000000000000000000000000000000e-01 4.87668880442191920998595833225409e-01 1.23311195578080790014041667745914e-02 2.48800419844458437293455155270294e-03
            4.10757909252076280903764882168616e-01 5.74710061408912786795610827539349e-01 1.45320293390108767894730590342078e-02 2.88495418493539936347080931966502e-03
            3.24384118273061905490806111629354e-01 6.58953681301480465037911926629022e-01 1.66622004254576294712819617416244e-02 3.14750131495105528456335264309018e-03
            2.43654731456761419750023378583137e-01 7.37692100676460293939840084931348e-01 1.86531678667782863101365364855155e-02 3.23084431280764965144225442372772e-03
            1.71164420391654581443674487672979e-01 8.08394638356514039223554846103070e-01 2.04409412518313793327706662239507e-02 3.10589531739593333048432555187901e-03
            1.09242998051599260467270369190373e-01 8.68788939772439983144636244105641e-01 2.19680621759608118992446179618128e-02 2.76358496010473131399010071618250e-03
            5.98804231365069883352703072887380e-02 9.16934123061613703953298681881279e-01 2.31854538018793077114310108299833e-02 2.21683240587659206172199688467117e-03
            2.46622391156161580028083335491829e-02 9.51283747806963564386251164251007e-01 2.40540130774202776109405021998100e-02 1.49995214231954496382492880712789e-03
            4.71226234279131794835393520770594e-03 9.70741713482266010970533898216672e-01 2.45460241749426710811121665756218e-02 6.66471299188430726413712168465509e-04
            9.95287737657208571029343602276640e-01 4.43009007977485016332996892174378e-03 2.82172263016578807326428801616203e-04 4.83826509084280636759697186155549e-06
            9.75337760884383841997191666450817e-01 2.31854538018793979170517616239522e-02 1.47678531373676008575657192523067e-03 5.81543584650973195064420362410118e-05
            9.40119576863492900642427230195608e-01 5.62947580614999651404950498090329e-02 3.58566507500713421707771999535908e-03 2.16502056708442475055700904285061e-04
            8.90757001948400684021578399551800e-01 1.02701481103568928454627950941358e-01 6.54151694803038752379364950684248e-03 5.19678305180139494569291347403350e-04
            8.28835579608345418556325512327021e-01 1.60915022472687324617268700421846e-01 1.02493979189672568264057872511330e-02 9.83465937782175688916530553740358e-04
            7.56345268543238580249976621416863e-01 2.29064583037918545471711695427075e-01 1.45901484188428742783116831560619e-02 1.59587248333772804657904043068584e-03
            6.75615881726938205531496350886300e-01 3.04959860012108063820335246418836e-01 1.94242582609537306481684026948642e-02 2.31714416114221300260767044676413e-03
            5.89242090747923774607386349089211e-01 3.86161551839394878715694403581438e-01 2.45963574126813466769192473293515e-02 3.08360324742398437744550498962326e-03
            5.00000000000000000000000000000000e-01 4.70059788431746450321213615097804e-01 2.99402115682535496787863849021960e-02 3.81485916705116357219851330739857e-03
            4.10757909252076280903764882168616e-01 5.53958025024097855393279132840689e-01 3.52840657238258081918047537328675e-02 4.42350295301101661959997102258058e-03
            3.24384118273061905490806111629354e-01 6.35159716851384725799789521261118e-01 4.04561648755533687094043671095278e-02 4.82606671329299449779703934382269e-03
            2.43654731456761419750023378583137e-01 7.11054993825574355170715534768533e-01 4.52902747176642250792610866483301e-02 4.95385660994249361899210981619035e-03
            1.71164420391654581443674487672979e-01 7.79204554390805603780734145402675e-01 4.96310252175398147755913669243455e-02 4.76227219828506426885228464129796e-03
            1.09242998051599260467270369190373e-01 8.37418095759924097087889549584361e-01 5.33389061884766979559913124830928e-02 4.23740740693733138055865339310913e-03
            5.98804231365069883352703072887380e-02 8.83824818801993039585340738995001e-01 5.62947580614999720793889537162613e-02 3.39907120360221758764707189470755e-03
            2.46622391156161580028083335491829e-02 9.16934123061613592930996219365625e-01 5.84036378227702490661954470851924e-02 2.29987802425856606783338698107855e-03
            4.71226234279131794835393520770594e-03 9.35689486783718105122886754543288e-01 5.95982508734905769287593102490064e-02 1.02190106707816829624546617338865e-03
            9.95287737657208571029343602276640e-01 4.19748067685924049208257002874234e-03 5.14781665932188478573827694617648e-04 6.36580704579817623127652517767139e-06
            9.75337760884383841997191666450817e-01 2.19680621759608743492897531268682e-02 2.69417693965528365351858042231470e-03 7.65149114217927395296264814206211e-05
            9.40119576863492900642427230195608e-01 5.33389061884767118337791202975495e-02 6.54151694803038752379364950684248e-03 2.84856305338226218053926386275521e-04
            8.90757001948400684021578399551800e-01 9.73089654282975891064211282355245e-02 1.19340326233017268720004722126760e-02 6.83751666051831872260302613852900e-04
            8.28835579608345418556325512327021e-01 1.52465905948305946449039538492798e-01 1.86985144433486349946349491801811e-02 1.29396679976989865436254323327603e-03
            7.56345268543238580249976621416863e-01 2.17037158102967475681310816071345e-01 2.66175733537939440687125625117915e-02 2.09972296016898906864667218030718e-03
            6.75615881726938205531496350886300e-01 2.88947424672587938054846290469868e-01 3.54366936004738564136573586438317e-02 3.04871526263554223912244367511448e-03
            5.89242090747923774607386349089211e-01 3.65885483771972608302291973814135e-01 4.48724254801036170903216770966537e-02 4.05716158795224948740987258588575e-03
            5.00000000000000000000000000000000e-01 4.45378500974200342010789199775900e-01 5.46214990257996579892108002241002e-02 5.01929036718237481934368204861130e-03
            4.10757909252076280903764882168616e-01 5.24871518176427964696983963222010e-01 6.43705725714956988880999233515467e-02 5.82009578571507195376977250589334e-03
            3.24384118273061905490806111629354e-01 6.01809577275812634944429646566277e-01 7.38063044511254595647642418043688e-02 6.34975738413311521779469615012204e-03
            2.43654731456761419750023378583137e-01 6.73719843845433152829116352222627e-01 8.26254246978054274208602691942360e-02 6.51789323638578711045399316503790e-03
            1.71164420391654581443674487672979e-01 7.38291096000094793083690092316829e-01 9.05444836082506254726354200101923e-02 6.26582160023211008625931839333134e-03
            1.09242998051599260467270369190373e-01 7.93448036520103205937459733831929e-01 9.73089654282975891064211282355245e-02 5.57524596534667883640512542342549e-03
            5.98804231365069883352703072887380e-02 8.37418095759924097087889549584361e-01 1.02701481103568914576840143126901e-01 4.47222940677925034913542390313523e-03
            2.46622391156161580028083335491829e-02 8.68788939772439872122333781589987e-01 1.06548821111943969874857884860830e-01 3.02599784352684938668054392962858e-03
            4.71226234279131794835393520770594e-03 8.86559521271541584042097383644432e-01 1.08728216385667098009548681147862e-01 1.34453670701654222525978088498277e-03
            9.95287737657208571029343602276640e-01 3.90569069015411383591351857091922e-03 8.06571652637315134742879152440764e-04 7.68879572985607591699072638213863e-06
            9.75337760884383841997191666450817e-01 2.04409412518313723938767623167223e-02 4.22129786378478560893157123246056e-03 9.24168011970320492321981364902683e-05
            9.40119576863492900642427230195608e-01 4.96310252175398355922730786460306e-02 1.02493979189672637652996911583614e-02 3.44057230819272213354770295623553e-04
            8.90757001948400684021578399551800e-01 9.05444836082506671059988434535626e-02 1.86985144433486488724227569946379e-02 8.25853949451923097409278273772770e-04
            8.28835579608345418556325512327021e-01 1.41867161583643514033425958587031e-01 2.92972588080110674102485290859477e-02 1.56288846536957351344987454666580e-03
            7.56345268543238580249976621416863e-01 2.01949710571280610649225195629697e-01 4.17050208854808091007981829534401e-02 2.53610277752360424238786684725255e-03
            6.75615881726938205531496350886300e-01 2.68861098684595234420413589759846e-01 5.55230195884665600480900593538536e-02 3.68232161676525747429078450068118e-03
            5.89242090747923774607386349089211e-01 3.40450769793656682349336506376858e-01 7.03071394584195430432771445339313e-02 4.90035064970650770937998785825584e-03
            5.00000000000000000000000000000000e-01 4.14417789804172709278162756163510e-01 8.55822101958272907218372438364895e-02 6.06243608460813650923304507500688e-03
            4.10757909252076280903764882168616e-01 4.88384809814688569673535312176682e-01 1.00857280933235093911548574396875e-01 7.02967075543026870265705952078861e-03
            3.24384118273061905490806111629354e-01 5.59974480923750128624760691309348e-01 1.15641400803187965884433197061298e-01 7.66941050985362436642756023275069e-03
            2.43654731456761419750023378583137e-01 6.26885869037064780151524701068411e-01 1.29459399506173800098451920348452e-01 7.87248958742154357481890514236511e-03
            1.71164420391654581443674487672979e-01 6.86968418024701876767323938111076e-01 1.41867161583643541789001574215945e-01 7.56803057606091761355671110322874e-03
            1.09242998051599260467270369190373e-01 7.38291096000094793083690092316829e-01 1.52465905948306001960190769750625e-01 6.73393445055009507577148397672318e-03
            5.98804231365069883352703072887380e-02 7.79204554390805714803036607918330e-01 1.60915022472687296861693084792932e-01 5.40168090524798232693326482944940e-03
            2.46622391156161580028083335491829e-02 8.08394638356514039223554846103070e-01 1.66943122527869802773636820347747e-01 3.65488289709002545091465208315640e-03
            4.71226234279131794835393520770594e-03 8.24929888918191411839586635323940e-01 1.70357848739017270212059429468354e-01 1.62396818143697410814774162446383e-03
            9.95287737657208571029343602276640e-01 3.56409732710477383452185584644667e-03 1.14816501568665513613454187691332e-03 8.76467529350308184772633257608732e-06
            9.75337760884383841997191666450817e-01 1.86531678667783071268182482072007e-02 6.00907124883785087599008534198219e-03 1.05348520446566060355941818471592e-04
            9.40119576863492900642427230195608e-01 4.52902747176642181403671827411017e-02 1.45901484188428812172055870632903e-02 3.92200549535110225968237429583496e-04
            8.90757001948400684021578399551800e-01 8.26254246978053719097090379364090e-02 2.66175733537939440687125625117915e-02 9.41414229369662920905714731389935e-04
            8.28835579608345418556325512327021e-01 1.29459399506173772342876304719539e-01 4.17050208854808091007981829534401e-02 1.78158067923883558884845523806462e-03
            7.56345268543238580249976621416863e-01 1.84287103295494886223693242754962e-01 5.93676281612665335263301358281751e-02 2.89097514577380454089960970520679e-03
            6.75615881726938205531496350886300e-01 2.45346393046400601534173802065197e-01 7.90377252266611929343298470485024e-02 4.19758235634631708677044770183784e-03
            5.89242090747923774607386349089211e-01 3.10674801179520787552945648712921e-01 1.00083108072555437839668002197868e-01 5.58604803379115881500904805534446e-03
            5.00000000000000000000000000000000e-01 3.78172634271619290124988310708432e-01 1.21827365728380709875011689291568e-01 6.91074202464227876907543191009609e-03
            4.10757909252076280903764882168616e-01 4.45670467363717681674728510188288e-01 1.43571623384205981910355376385269e-01 8.01332012922836671442450295899107e-03
            3.24384118273061905490806111629354e-01 5.10998875496837867693500356836012e-01 1.64617006230100226815693531534635e-01 8.74257753401197316989179597612747e-03
            2.43654731456761419750023378583137e-01 5.72058165247743666270707763032988e-01 1.84287103295494913979268858383875e-01 8.97407310709312511232127462790231e-03
            1.71164420391654581443674487672979e-01 6.26885869037064780151524701068411e-01 2.01949710571280638404800811258610e-01 8.62701168570629249587877751537235e-03
            1.09242998051599260467270369190373e-01 6.73719843845433263851418814738281e-01 2.17037158102967531192462047329172e-01 7.67620196718510686140213294947898e-03
            5.98804231365069883352703072887380e-02 7.11054993825574466193017997284187e-01 2.29064583037918545471711695427075e-01 6.15752854374511555957116826220954e-03
            2.46622391156161580028083335491829e-02 7.37692100676460293939840084931348e-01 2.37645660207923548057351581519470e-01 4.16630418524222348453101005816279e-03
            4.71226234279131794835393520770594e-03 7.52781171216133859758201651857235e-01 2.42506566441074822293444412935060e-01 1.85120717175590865366530124447308e-03
            9.95287737657208571029343602276640e-01 3.18367927765367871723034554065634e-03 1.52858306513775025342605218270364e-03 9.55885701160958090830555705652571e-06
            9.75337760884383841997191666450817e-01 1.66622004254576329407289136952386e-02 8.00003869015852506207941985394427e-03 1.14894324046415339819066092630351e-04
            9.40119576863492900642427230195608e-01 4.04561648755533964649799827384413e-02 1.94242582609537028925927870659507e-02 4.27738489714479407002162281159485e-04
            8.90757001948400684021578399551800e-01 7.38063044511254595647642418043688e-02 3.54366936004738564136573586438317e-02 1.02671732903895853057651965656305e-03
            8.28835579608345418556325512327021e-01 1.15641400803188035273372236133582e-01 5.55230195884665461703022515393968e-02 1.94301264989404681889084702817172e-03
            7.56345268543238580249976621416863e-01 1.64617006230100226815693531534635e-01 7.90377252266611929343298470485024e-02 3.15293118309280685079842854179333e-03
            6.75615881726938205531496350886300e-01 2.19159062085270056607910760249069e-01 1.05225056187791737860592888864630e-01 4.57793223309848305857538974805720e-03
            5.89242090747923774607386349089211e-01 2.77514567035655135285310279869009e-01 1.33243342216421090107303371041780e-01 6.09220908098821734011352191373589e-03
            5.00000000000000000000000000000000e-01 3.37807940863469102765748175443150e-01 1.62192059136530897234251824556850e-01 7.53693578433461269794335635197058e-03
            4.10757909252076280903764882168616e-01 3.98101314691282959223883608501637e-01 1.91140776056640704361200278071919e-01 8.73942031375949356097887488203924e-03
            3.24384118273061905490806111629354e-01 4.56456819641668065656858743750490e-01 2.19159062085270028852335144620156e-01 9.53475693759896734957859365522381e-03
            2.43654731456761419750023378583137e-01 5.10998875496837978715802819351666e-01 2.45346393046400601534173802065197e-01 9.78722870726550855935421679987485e-03
            1.71164420391654581443674487672979e-01 5.59974480923750128624760691309348e-01 2.68861098684595289931564821017673e-01 9.40871947672483437130619421395750e-03
            1.09242998051599260467270369190373e-01 6.01809577275812856989034571597585e-01 2.88947424672587938054846290469868e-01 8.37175531772970868416017253821337e-03
            5.98804231365069883352703072887380e-02 6.35159716851384836822091983776772e-01 3.04959860012108174842637708934490e-01 6.71547238472074495274632965902128e-03
            2.46622391156161580028083335491829e-02 6.58953681301480576060214389144676e-01 3.16384079582903265936977277306141e-01 4.54381989520157188178206553175187e-03
            4.71226234279131794835393520770594e-03 6.72432202449284588396949402522296e-01 3.22855535207924093654696662269998e-01 2.01894811400461738834510860840510e-03
            9.95287737657208571029343602276640e-01 2.77666331501913096757583332419017e-03 1.93559902777229800308056439916982e-03 1.00458121707907386269608904227901e-05
            9.75337760884383841997191666450817e-01 1.45320293390109236270069104079994e-02 1.01302097766052343758014231411835e-02 1.20747365240261240646423013256339e-04
            9.40119576863492900642427230195608e-01 3.52840657238257873751230420111824e-02 2.45963574126813119824497277932096e-02 4.49528695812744807057403928496342e-04
            8.90757001948400684021578399551800e-01 6.43705725714957405214633467949170e-02 4.48724254801035754569582536532835e-02 1.07902120802668604129803675562016e-03
            8.28835579608345418556325512327021e-01 1.00857280933235107789336382211332e-01 7.03071394584194736543381054616475e-02 2.04199519907027286086398376596662e-03
            7.56345268543238580249976621416863e-01 1.43571623384206009665930992014182e-01 1.00083108072555410084092386568955e-01 3.31355039774215969633486267298395e-03
            6.75615881726938205531496350886300e-01 1.91140776056640704361200278071919e-01 1.33243342216421090107303371041780e-01 4.81114502376807102318467812551717e-03
            5.89242090747923774607386349089211e-01 2.42035849238939310135165783322009e-01 1.68722060013136915257447867588780e-01 6.40256340883248002693317602052048e-03
            5.00000000000000000000000000000000e-01 2.94621045373961887303693174544605e-01 2.05378954626038112696306825455395e-01 7.92088856866241322607358199547889e-03
            4.10757909252076280903764882168616e-01 3.47206241508984381205493718880462e-01 2.42035849238939282379590167693095e-01 9.18463105442326663285790289137367e-03
            3.24384118273061905490806111629354e-01 3.98101314691283014735034839759464e-01 2.77514567035655079774159048611182e-01 1.00204843709796580925042164267325e-02
            2.43654731456761419750023378583137e-01 4.45670467363717737185879741446115e-01 3.10674801179520843064096879970748e-01 1.02858177652774083338238497731254e-02
            1.71164420391654581443674487672979e-01 4.88384809814688680695837774692336e-01 3.40450769793656737860487737634685e-01 9.88802620606651569423739545072749e-03
            1.09242998051599260467270369190373e-01 5.24871518176428075719286425737664e-01 3.65885483771972719324594436329789e-01 8.79823616564065838496766502885293e-03
            5.98804231365069883352703072887380e-02 5.53958025024098077437884057871997e-01 3.86161551839394934226845634839265e-01 7.05757750462228383714347046407056e-03
            2.46622391156161580028083335491829e-02 5.74710061408912897817913290055003e-01 4.00627699475470944179278376395814e-01 4.77529490708542610905018577227565e-03
            4.71226234279131794835393520770594e-03 5.86465427432904751192666026327061e-01 4.08822310224303930858980038465234e-01 2.12179903007539459938723958032369e-03
            9.95287737657208571029343602276640e-01 2.35613117139571448532819886167999e-03 2.35613117139571448532819886167999e-03 1.02098885048394470053323077052276e-05
            9.75337760884383841997191666450817e-01 1.23311195578081067569797824035049e-02 1.23311195578080512458285511456779e-02 1.22719508925394775499320698664008e-04
            9.40119576863492900642427230195608e-01 2.99402115682535496787863849021960e-02 2.99402115682535496787863849021960e-02 4.56870762258413407128770922227545e-04
            8.90757001948400684021578399551800e-01 5.46214990257996579892108002241002e-02 5.46214990257996579892108002241002e-02 1.09664465560503068472519849763103e-03
            8.28835579608345418556325512327021e-01 8.55822101958272907218372438364895e-02 8.55822101958272907218372438364895e-02 2.07534671716680732544979548492847e-03
            7.56345268543238580249976621416863e-01 1.21827365728380709875011689291568e-01 1.21827365728380709875011689291568e-01 3.36766998436234012148648986340049e-03
            6.75615881726938205531496350886300e-01 1.62192059136530897234251824556850e-01 1.62192059136530897234251824556850e-01 4.88972453776414118370441386218772e-03
            5.89242090747923774607386349089211e-01 2.05378954626038084940731209826481e-01 2.05378954626038140451882441084308e-01 6.50713525576488652918394350876952e-03
            5.00000000000000000000000000000000e-01 2.50000000000000000000000000000000e-01 2.50000000000000000000000000000000e-01 8.05025893082519233834304372976476e-03
            4.10757909252076280903764882168616e-01 2.94621045373961831792541943286778e-01 2.94621045373961831792541943286778e-01 9.33464188155991130002586686487120e-03
            3.24384118273061905490806111629354e-01 3.37807940863469047254596944185323e-01 3.37807940863469047254596944185323e-01 1.01841470309050729364797049925073e-02
            2.43654731456761419750023378583137e-01 3.78172634271619290124988310708432e-01 3.78172634271619290124988310708432e-01 1.04538140649222347638991337248626e-02
            1.71164420391654581443674487672979e-01 4.14417789804172709278162756163510e-01 4.14417789804172709278162756163510e-01 1.00495254520494570193989147810498e-02
            1.09242998051599260467270369190373e-01 4.45378500974200397521940431033727e-01 4.45378500974200397521940431033727e-01 8.94193607875972372445172453581108e-03
            5.98804231365069883352703072887380e-02 4.70059788431746505832364846355631e-01 4.70059788431746505832364846355631e-01 7.17284757184392766926617213130157e-03
            2.46622391156161580028083335491829e-02 4.87668880442191920998595833225409e-01 4.87668880442191920998595833225409e-01 4.85328888796376575753832582904579e-03
            4.71226234279131794835393520770594e-03 4.97643868828604341025823032396147e-01 4.97643868828604341025823032396147e-01 2.15645396892196394497687172986389e-03
            9.95287737657208571029343602276640e-01 1.93559902777229821992099889627070e-03 2.77666331501913096757583332419017e-03 1.00458121707907098278406837765786e-05
            9.75337760884383841997191666450817e-01 1.01302097766052898869526543990105e-02 1.45320293390108681158556791501724e-02 1.20747365240260888280716955467398e-04
            9.40119576863492900642427230195608e-01 2.45963574126813119824497277932096e-02 3.52840657238257873751230420111824e-02 4.49528695812743451804688321615799e-04
            8.90757001948400684021578399551800e-01 4.48724254801035754569582536532835e-02 6.43705725714957405214633467949170e-02 1.07902120802668278869151929910686e-03
            8.28835579608345418556325512327021e-01 7.03071394584194875321259132761043e-02 1.00857280933235093911548574396875e-01 2.04199519907026678933181784714179e-03
            7.56345268543238580249976621416863e-01 1.00083108072555423961880194383411e-01 1.43571623384205981910355376385269e-01 3.31355039774214972167487580634315e-03
            6.75615881726938205531496350886300e-01 1.33243342216421090107303371041780e-01 1.91140776056640704361200278071919e-01 4.81114502376805627803513232265686e-03
            5.89242090747923774607386349089211e-01 1.68722060013136887501872251959867e-01 2.42035849238939337890741398950922e-01 6.40256340883246094497494027564244e-03
            5.00000000000000000000000000000000e-01 2.05378954626038140451882441084308e-01 2.94621045373961831792541943286778e-01 7.92088856866239067466839429698666e-03
            4.10757909252076280903764882168616e-01 2.42035849238939337890741398950922e-01 3.47206241508984325694342487622635e-01 9.18463105442323887728228726246016e-03
            3.24384118273061905490806111629354e-01 2.77514567035655135285310279869009e-01 3.98101314691282959223883608501637e-01 1.00204843709796286022051248210119e-02
            2.43654731456761419750023378583137e-01 3.10674801179520843064096879970748e-01 4.45670467363717737185879741446115e-01 1.02858177652773788435247581674048e-02
            1.71164420391654581443674487672979e-01 3.40450769793656793371638968892512e-01 4.88384809814688625184686543434509e-01 9.88802620606648620393830384500689e-03
            1.09242998051599260467270369190373e-01 3.65885483771972774835745667587616e-01 5.24871518176428075719286425737664e-01 8.79823616564063236411552537674652e-03
            5.98804231365069883352703072887380e-02 3.86161551839394989737996866097092e-01 5.53958025024098077437884057871997e-01 7.05757750462226302046175874238543e-03
            2.46622391156161580028083335491829e-02 4.00627699475470999690429607653641e-01 5.74710061408912897817913290055003e-01 4.77529490708541223126237795781890e-03
            4.71226234279131794835393520770594e-03 4.08822310224304041881282500980888e-01 5.86465427432904640170363563811406e-01 2.12179903007538809417420466729709e-03
            9.95287737657208571029343602276640e-01 1.52858306513775068710692117690542e-03 3.18367927765367828354947654645457e-03 9.55885701160957413204197902212300e-06
            9.75337760884383841997191666450817e-01 8.00003869015858404267760306538548e-03 1.66622004254575739601307304837974e-02 1.14894324046415244951376000148713e-04
            9.40119576863492900642427230195608e-01 1.94242582609537098314866909731791e-02 4.04561648755533895260860788312129e-02 4.27738489714479027531401911232933e-04
            8.90757001948400684021578399551800e-01 3.54366936004738633525512625510601e-02 7.38063044511254595647642418043688e-02 1.02671732903895766321478166815950e-03
            8.28835579608345418556325512327021e-01 5.55230195884665669869839632610820e-02 1.15641400803188021395584428319125e-01 1.94301264989404508416737105136463e-03
            7.56345268543238580249976621416863e-01 7.90377252266612345676932704918727e-02 1.64617006230100171304542300276808e-01 3.15293118309280424871321457658269e-03
            6.75615881726938205531496350886300e-01 1.05225056187791779493956312308001e-01 2.19159062085270028852335144620156e-01 4.57793223309847958912843779444302e-03
            5.89242090747923774607386349089211e-01 1.33243342216421117862878986670694e-01 2.77514567035655135285310279869009e-01 6.09220908098821213594309398331461e-03
            5.00000000000000000000000000000000e-01 1.62192059136530952745403055814677e-01 3.37807940863469047254596944185323e-01 7.53693578433460662641119043314575e-03
            4.10757909252076280903764882168616e-01 1.91140776056640759872351509329746e-01 3.98101314691282903712732377243810e-01 8.73942031375948662208497097481086e-03
            3.24384118273061905490806111629354e-01 2.19159062085270084363486375877983e-01 4.56456819641668010145707512492663e-01 9.53475693759895867596121377118834e-03
            2.43654731456761419750023378583137e-01 2.45346393046400684800900648951938e-01 5.10998875496837867693500356836012e-01 9.78722870726549988573683691583938e-03
            1.71164420391654581443674487672979e-01 2.68861098684595345442716052275500e-01 5.59974480923750128624760691309348e-01 9.40871947672482569768881432992202e-03
            1.09242998051599260467270369190373e-01 2.88947424672588104588299984243349e-01 6.01809577275812745966732109081931e-01 8.37175531772970347998974460779209e-03
            5.98804231365069883352703072887380e-02 3.04959860012108230353788940192317e-01 6.35159716851384725799789521261118e-01 6.71547238472073974857590172860000e-03
            2.46622391156161580028083335491829e-02 3.16384079582903321448128508563968e-01 6.58953681301480465037911926629022e-01 4.54381989520156841233511357813768e-03
            4.71226234279131794835393520770594e-03 3.22855535207924204676999124785652e-01 6.72432202449284477374646940006642e-01 2.01894811400461565362163263159800e-03
            9.95287737657208571029343602276640e-01 1.14816501568665513613454187691332e-03 3.56409732710477383452185584644667e-03 8.76467529350309540025348864489274e-06
            9.75337760884383841997191666450817e-01 6.00907124883790551977957861140567e-03 1.86531678667782516156670169493736e-02 1.05348520446566236538794847366063e-04
            9.40119576863492900642427230195608e-01 1.45901484188428812172055870632903e-02 4.52902747176642181403671827411017e-02 3.92200549535110822279432296610935e-04
            8.90757001948400684021578399551800e-01 2.66175733537939440687125625117915e-02 8.26254246978053719097090379364090e-02 9.41414229369664438788756211096143e-04
            8.28835579608345418556325512327021e-01 4.17050208854808160396920868606685e-02 1.29459399506173772342876304719539e-01 1.78158067923883840777410370037614e-03
            7.56345268543238580249976621416863e-01 5.93676281612665196485423280137184e-02 1.84287103295494913979268858383875e-01 2.89097514577380887770829964722452e-03
            6.75615881726938205531496350886300e-01 7.90377252266612068121176548629592e-02 2.45346393046400601534173802065197e-01 4.19758235634632402566435160906622e-03
            5.89242090747923774607386349089211e-01 1.00083108072555396206304578754498e-01 3.10674801179520843064096879970748e-01 5.58604803379116835598816592778348e-03
            5.00000000000000000000000000000000e-01 1.21827365728380709875011689291568e-01 3.78172634271619290124988310708432e-01 6.91074202464229004477802575934220e-03
            4.10757909252076280903764882168616e-01 1.43571623384205981910355376385269e-01 4.45670467363717681674728510188288e-01 8.01332012922838059221231077344783e-03
            3.24384118273061905490806111629354e-01 1.64617006230100199060117915905721e-01 5.10998875496837867693500356836012e-01 8.74257753401198878240307976739132e-03
            2.43654731456761419750023378583137e-01 1.84287103295494886223693242754962e-01 5.72058165247743666270707763032988e-01 8.97407310709313899010908244235907e-03
            1.71164420391654581443674487672979e-01 2.01949710571280610649225195629697e-01 6.26885869037064780151524701068411e-01 8.62701168570630637366658532982910e-03
            1.09242998051599260467270369190373e-01 2.17037158102967503436886431700259e-01 6.73719843845433263851418814738281e-01 7.67620196718511900446646478712864e-03
            5.98804231365069883352703072887380e-02 2.29064583037918573227287311055989e-01 7.11054993825574466193017997284187e-01 6.15752854374512510055028613464856e-03
            2.46622391156161580028083335491829e-02 2.37645660207923520301775965890556e-01 7.37692100676460293939840084931348e-01 4.16630418524223042342491396539117e-03
            4.71226234279131794835393520770594e-03 2.42506566441074794537868797306146e-01 7.52781171216133859758201651857235e-01 1.85120717175591168943138420388550e-03
            9.95287737657208571029343602276640e-01 8.06571652637315243163096400991208e-04 3.90569069015411383591351857091922e-03 7.68879572985606914072714834773592e-06
            9.75337760884383841997191666450817e-01 4.22129786378484112008280249028758e-03 2.04409412518313168827255310588953e-02 9.24168011970319543645080440086303e-05
            9.40119576863492900642427230195608e-01 1.02493979189672602958527392047472e-02 4.96310252175398425311669825532590e-02 3.44057230819271888094118549972222e-04
            8.90757001948400684021578399551800e-01 1.86985144433486488724227569946379e-02 9.05444836082506671059988434535626e-02 8.25853949451922338467757533919666e-04
            8.28835579608345418556325512327021e-01 2.92972588080110570019076732251051e-02 1.41867161583643514033425958587031e-01 1.56288846536957199556683306695959e-03
            7.56345268543238580249976621416863e-01 4.17050208854808160396920868606685e-02 2.01949710571280610649225195629697e-01 2.53610277752360164030265288204191e-03
            6.75615881726938205531496350886300e-01 5.55230195884665531091961554466252e-02 2.68861098684595234420413589759846e-01 3.68232161676525357116296355286522e-03
            5.89242090747923774607386349089211e-01 7.03071394584194736543381054616475e-02 3.40450769793656737860487737634685e-01 4.90035064970650250520955992783456e-03
            5.00000000000000000000000000000000e-01 8.55822101958272907218372438364895e-02 4.14417789804172709278162756163510e-01 6.06243608460813043770087915618205e-03
            4.10757909252076280903764882168616e-01 1.00857280933235080033760766582418e-01 4.88384809814688569673535312176682e-01 7.02967075543026176376315561356023e-03
            3.24384118273061905490806111629354e-01 1.15641400803188007517796620504669e-01 5.59974480923750128624760691309348e-01 7.66941050985361656017191833711877e-03
            2.43654731456761419750023378583137e-01 1.29459399506173772342876304719539e-01 6.26885869037064780151524701068411e-01 7.87248958742153663592500123513673e-03
            1.71164420391654581443674487672979e-01 1.41867161583643514033425958587031e-01 6.86968418024701876767323938111076e-01 7.56803057606090980730106920759681e-03
            1.09242998051599260467270369190373e-01 1.52465905948305946449039538492798e-01 7.38291096000094793083690092316829e-01 6.73393445055008813687758006949480e-03
            5.98804231365069883352703072887380e-02 1.60915022472687352372844316050760e-01 7.79204554390805714803036607918330e-01 5.40168090524797712276283689902812e-03
            2.46622391156161580028083335491829e-02 1.66943122527869747262485589089920e-01 8.08394638356514150245857308618724e-01 3.65488289709002198146770012954221e-03
            4.71226234279131794835393520770594e-03 1.70357848739017297967635045097268e-01 8.24929888918191411839586635323940e-01 1.62396818143697237342426564765674e-03
            9.95287737657208571029343602276640e-01 5.14781665932188695414262191718535e-04 4.19748067685924049208257002874234e-03 6.36580704579817877237536694057241e-06
            9.75337760884383841997191666450817e-01 2.69417693965533699626546670913285e-03 2.19680621759608223075854738226553e-02 7.65149114217927666346807935582319e-05
            9.40119576863492900642427230195608e-01 6.54151694803038405434669755322830e-03 5.33389061884767118337791202975495e-02 2.84856305338226326474143634825964e-04
            8.90757001948400684021578399551800e-01 1.19340326233017251372769962358689e-02 9.73089654282975891064211282355245e-02 6.83751666051832089100737110953787e-04
            8.28835579608345418556325512327021e-01 1.86985144433486384640819011337953e-02 1.52465905948305946449039538492798e-01 1.29396679976989908804341222747780e-03
            7.56345268543238580249976621416863e-01 2.66175733537939301909247546973347e-02 2.17037158102967475681310816071345e-01 2.09972296016898993600841016871072e-03
            6.75615881726938205531496350886300e-01 3.54366936004738355969756469221466e-02 2.88947424672587938054846290469868e-01 3.04871526263554354016505065771980e-03
            5.89242090747923774607386349089211e-01 4.48724254801035407624887341171416e-02 3.65885483771972663813443205071962e-01 4.05716158795225122213334856269285e-03
            5.00000000000000000000000000000000e-01 5.46214990257996302336351845951867e-02 4.45378500974200397521940431033727e-01 5.01929036718237742142889601382194e-03
            4.10757909252076280903764882168616e-01 6.43705725714956988880999233515467e-02 5.24871518176427964696983963222010e-01 5.82009578571507368849324848270044e-03
            3.24384118273061905490806111629354e-01 7.38063044511254179314008183609985e-02 6.01809577275812634944429646566277e-01 6.34975738413311695251817212692913e-03
            2.43654731456761419750023378583137e-01 8.26254246978053302763456144930387e-02 6.73719843845433263851418814738281e-01 6.51789323638578884517746914184499e-03
            1.71164420391654581443674487672979e-01 9.05444836082506254726354200101923e-02 7.38291096000094793083690092316829e-01 6.26582160023211268834453235854198e-03
            1.09242998051599260467270369190373e-01 9.73089654282975474730577047921543e-02 7.93448036520103205937459733831929e-01 5.57524596534668057112860140023258e-03
            5.98804231365069883352703072887380e-02 1.02701481103568886821264527497988e-01 8.37418095759924097087889549584361e-01 4.47222940677925208385889987994233e-03
            2.46622391156161580028083335491829e-02 1.06548821111943928241494461417460e-01 8.68788939772439872122333781589987e-01 3.02599784352685025404228191803213e-03
            4.71226234279131794835393520770594e-03 1.08728216385667084131760873333405e-01 8.86559521271541584042097383644432e-01 1.34453670701654265894064987918455e-03
            9.95287737657208571029343602276640e-01 2.82172263016578482065777055964872e-04 4.43009007977485016332996892174378e-03 4.83826509084281568495939165885922e-06
            9.75337760884383841997191666450817e-01 1.47678531373681277798215472074617e-03 2.31854538018793458753474823197394e-02 5.81543584650974279266592847914552e-05
            9.40119576863492900642427230195608e-01 3.58566507500712814554555407653424e-03 5.62947580614999720793889537162613e-02 2.16502056708442854526461274211613e-04
            8.90757001948400684021578399551800e-01 6.54151694803037538072931766919282e-03 1.02701481103568942332415758755815e-01 5.19678305180140470351246584357341e-04
            8.28835579608345418556325512327021e-01 1.02493979189672412138945034598692e-02 1.60915022472687352372844316050760e-01 9.83465937782177423640006530547453e-04
            7.56345268543238580249976621416863e-01 1.45901484188428534616299714343768e-02 2.29064583037918573227287311055989e-01 1.59587248333773086550468889299736e-03
            6.75615881726938205531496350886300e-01 1.94242582609536681981232675298088e-02 3.04959860012108119331486477676663e-01 2.31714416114221733941636038878187e-03
            5.89242090747923774607386349089211e-01 2.45963574126812599407454484889968e-02 3.86161551839394989737996866097092e-01 3.08360324742399001529680191424632e-03
            5.00000000000000000000000000000000e-01 2.99402115682534941676351536443690e-02 4.70059788431746505832364846355631e-01 3.81485916705117051109241721462695e-03
            4.10757909252076280903764882168616e-01 3.52840657238257179861840029388986e-02 5.53958025024097966415581595356343e-01 4.42350295301102442585561291821250e-03
            3.24384118273061905490806111629354e-01 4.04561648755533131982531358517008e-02 6.35159716851384725799789521261118e-01 4.82606671329300317141441922785816e-03
            2.43654731456761419750023378583137e-01 4.52902747176641348736403358543612e-02 7.11054993825574466193017997284187e-01 4.95385660994250315997122768862937e-03
            1.71164420391654581443674487672979e-01 4.96310252175397453866523278520617e-02 7.79204554390805714803036607918330e-01 4.76227219828507294246966452533343e-03
            1.09242998051599260467270369190373e-01 5.33389061884766216281583695035806e-02 8.37418095759924208110192012100015e-01 4.23740740693733831945255730033750e-03
            5.98804231365069883352703072887380e-02 5.62947580614998679959803951078356e-02 8.83824818801993150607643201510655e-01 3.39907120360222365917923781353238e-03
            2.46622391156161580028083335491829e-02 5.84036378227701796772564080129087e-02 9.16934123061613703953298681881279e-01 2.29987802425857040464207692309628e-03
            4.71226234279131794835393520770594e-03 5.95982508734904173341995203827537e-02 9.35689486783718216145189217058942e-01 1.02190106707817024780937664729663e-03
            9.95287737657208571029343602276640e-01 1.16214940673435815116261438095790e-04 4.59604740211799322330277206560822e-03 3.15545694666093726719630768728919e-06
            9.75337760884383841997191666450817e-01 6.08226038195829000684855536462692e-04 2.40540130774203296526447815040228e-02 3.79275568725268082018002580024785e-05
            9.40119576863492900642427230195608e-01 1.47678531373681212746085122944351e-03 5.84036378227702906995588705285627e-02 1.41199976847079030229520357764272e-04
            8.90757001948400684021578399551800e-01 2.69417693965533222577590777291334e-03 1.06548821111943983752645692675287e-01 3.38927794843916477176420931627376e-04
            8.28835579608345418556325512327021e-01 4.22129786378483157910368461784856e-03 1.66943122527869747262485589089920e-01 6.41404380891126297409288170570107e-04
            7.56345268543238580249976621416863e-01 6.00907124883789250935350878535246e-03 2.37645660207923520301775965890556e-01 1.04080839288115041020577056229968e-03
            6.75615881726938205531496350886300e-01 8.00003869015856322599589134370035e-03 3.16384079582903210425826046048314e-01 1.51121290429693383439158083092480e-03
            5.89242090747923774607386349089211e-01 1.01302097766052656008239907237112e-02 4.00627699475470944179278376395814e-01 2.01108808739028366296963667991804e-03
            5.00000000000000000000000000000000e-01 1.23311195578080790014041667745914e-02 4.87668880442191920998595833225409e-01 2.48800419844458046980673060488698e-03
            4.10757909252076280903764882168616e-01 1.45320293390108889325373908718575e-02 5.74710061408912786795610827539349e-01 2.88495418493539459298125038344551e-03
            3.24384118273061905490806111629354e-01 1.66622004254575913073654902518683e-02 6.58953681301480465037911926629022e-01 3.14750131495105008039292471266890e-03
            2.43654731456761419750023378583137e-01 1.86531678667782654934548247638304e-02 7.37692100676460293939840084931348e-01 3.23084431280764444727182649330643e-03
            1.71164420391654581443674487672979e-01 2.04409412518313272910663869197379e-02 8.08394638356514039223554846103070e-01 3.10589531739592812631389762145773e-03
            1.09242998051599260467270369190373e-01 2.19680621759608292464793777298837e-02 8.68788939772439983144636244105641e-01 2.76358496010472697718141077416476e-03
            5.98804231365069883352703072887380e-02 2.31854538018793493447944342733535e-02 9.16934123061613703953298681881279e-01 2.21683240587658859227504493105698e-03
            2.46622391156161580028083335491829e-02 2.40540130774203296526447815040228e-02 9.51283747806963564386251164251007e-01 1.49995214231954257858014933901813e-03
            4.71226234279131794835393520770594e-03 2.45460241749427265922633978334488e-02 9.70741713482266010970533898216672e-01 6.66471299188429642211539682961075e-04
            9.95287737657208571029343602276640e-01 2.22054163872896432100050717606265e-05 4.69005692640413890392991902444919e-03 1.37395558340913239250315192041541e-06
            9.75337760884383841997191666450817e-01 1.16214940673433335003791877504398e-04 2.45460241749427231228164458798346e-02 1.65144951780180537694725506625559e-05
            9.40119576863492900642427230195608e-01 2.82172263016572356323502512864820e-04 5.95982508734905283565019828984077e-02 6.14815856611750907517385811296151e-05
            8.90757001948400684021578399551800e-01 5.14781665932176769190364851169761e-04 1.08728216385667139642912104591232e-01 1.47576640711612845288658313336327e-04
            8.28835579608345418556325512327021e-01 8.06571652637296269625077904663613e-04 1.70357848739017297967635045097268e-01 2.79281620774759651790186687492223e-04
            7.56345268543238580249976621416863e-01 1.14816501568662803108022973930247e-03 2.42506566441074794537868797306146e-01 4.53190940909960909583481525331194e-04
            6.75615881726938205531496350886300e-01 1.52858306513771404107349116685555e-03 3.22855535207924093654696662269998e-01 6.58015445203858012970798974095032e-04
            5.89242090747923774607386349089211e-01 1.93559902777225203290845101378181e-03 4.08822310224303986370131269723061e-01 8.75672130250865080394739869973364e-04
            5.00000000000000000000000000000000e-01 2.35613117139565897417696760385297e-03 4.97643868828604341025823032396147e-01 1.08333192871340141562319647761115e-03
            4.10757909252076280903764882168616e-01 2.77666331501906504808374620552058e-03 5.86465427432904640170363563811406e-01 1.25617271199531584796593719488556e-03
            3.24384118273061905490806111629354e-01 3.18367927765360325675914054954774e-03 6.72432202449284477374646940006642e-01 1.37049152581236732828529056860134e-03
            2.43654731456761419750023378583137e-01 3.56409732710469013411413996550436e-03 7.52781171216133859758201651857235e-01 1.40678090613945280459851083776357e-03
            1.71164420391654581443674487672979e-01 3.90569069015402189556929180014322e-03 8.24929888918191411839586635323940e-01 1.35237535639206891274755051313150e-03
            1.09242998051599260467270369190373e-01 4.19748067685914161284443935073796e-03 8.86559521271541695064399846160086e-01 1.20332587335072798903623514377159e-03
            5.98804231365069883352703072887380e-02 4.43009007977474607992141031331812e-03 9.35689486783718216145189217058942e-01 9.65257746507837714397548545264272e-04
            2.46622391156161580028083335491829e-02 4.59604740211788480308552351516482e-03 9.70741713482266010970533898216672e-01 6.53112260957072429147507452995569e-04
            4.71226234279131794835393520770594e-03 4.69005692640402874898919449719870e-03 9.90597680730804674986700320005184e-01 2.90196310132155306082263335198945e-04
            ];
        
    case  32
        
        % ALG. DEG.:   32
        % PTS CARD.:  289
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.1e-15
        
        xyw_bar=[
            9.95287737657208571029343602276640e-01 4.69005692640413890392991902444919e-03 2.22054163872900667264786989107961e-05 1.37395558340911968700894310591032e-06
            9.75337760884383841997191666450817e-01 2.45460241749427786339676771376617e-02 1.16214940673379368840656411521195e-04 1.65144951780179013035420448884949e-05
            9.40119576863492900642427230195608e-01 5.95982508734905214176080789911794e-02 2.82172263016577939964690813212655e-04 6.14815856611745350981251823085927e-05
            8.90757001948400684021578399551800e-01 1.08728216385667125765124296776776e-01 5.14781665932190213297303671424743e-04 1.47576640711611462930888394318174e-04
            8.28835579608345418556325512327021e-01 1.70357848739017270212059429468354e-01 8.06571652637311231615058204624802e-04 2.79281620774757049704972722281582e-04
            7.56345268543238580249976621416863e-01 2.42506566441074766782293181677232e-01 1.14816501568665296773019690590445e-03 4.53190940909956735405117456139124e-04
            6.75615881726938205531496350886300e-01 3.22855535207924038143545431012171e-01 1.52858306513775632495821810152847e-03 6.58015445203851941438633055270202e-04
            5.89242090747923774607386349089211e-01 4.08822310224303875347828807207406e-01 1.93559902777235004478484370338265e-03 8.75672130250856948878446228690109e-04
            5.00000000000000000000000000000000e-01 4.97643868828604285514671801138320e-01 2.35613117139571448532819886167999e-03 1.08333192871339144096320961097035e-03
            4.10757909252076280903764882168616e-01 5.86465427432904529148061101295752e-01 2.77666331501913443702278527780436e-03 1.25617271199530413858247435143767e-03
            3.24384118273061905490806111629354e-01 6.72432202449284366352344477490988e-01 3.18367927765372815684941087965854e-03 1.37049152581235475154008973674991e-03
            2.43654731456761419750023378583137e-01 7.52781171216133859758201651857235e-01 3.56409732710472049177496955962852e-03 1.40678090613944001101287550881125e-03
            1.71164420391654581443674487672979e-01 8.24929888918191300817284172808286e-01 3.90569069015411773904133951873519e-03 1.35237535639205655284278417838095e-03
            1.09242998051599260467270369190373e-01 8.86559521271541584042097383644432e-01 4.19748067685921100178347842302173e-03 1.20332587335071693017407579162636e-03
            5.98804231365069883352703072887380e-02 9.35689486783718216145189217058942e-01 4.43009007977479551954047565232031e-03 9.65257746507828823939734164127913e-04
            2.46622391156161580028083335491829e-02 9.70741713482265899948231435701018e-01 4.59604740211794204896023074979894e-03 6.53112260957066466035558782721182e-04
            4.71226234279131794835393520770594e-03 9.90597680730804563964397857489530e-01 4.69005692640411808724820730276406e-03 2.90196310132152649786940745713082e-04
            9.95287737657208571029343602276640e-01 4.59604740211799322330277206560822e-03 1.16214940673435747353625657751763e-04 3.15545694666094234939399121309123e-06
            9.75337760884383841997191666450817e-01 2.40540130774203851637960127618499e-02 6.08226038195772839012320787333010e-04 3.79275568725268691881724603121029e-05
            9.40119576863492900642427230195608e-01 5.84036378227702837606649666213343e-02 1.47678531373681559690780318305769e-03 1.41199976847079247069954854865159e-04
            8.90757001948400684021578399551800e-01 1.06548821111943983752645692675287e-01 2.69417693965533222577590777291334e-03 3.38927794843917019277507174379593e-04
            8.28835579608345418556325512327021e-01 1.66943122527869747262485589089920e-01 4.22129786378483418118889858305920e-03 6.41404380891127381611460656074541e-04
            7.56345268543238580249976621416863e-01 2.37645660207923520301775965890556e-01 6.00907124883789944824741269258084e-03 1.04080839288115214492924653910677e-03
            6.75615881726938205531496350886300e-01 3.16384079582903210425826046048314e-01 8.00003869015858404267760306538548e-03 1.51121290429693621963636029903455e-03
            5.89242090747923774607386349089211e-01 4.00627699475470888668127145137987e-01 1.01302097766053367244865057728020e-02 2.01108808739028669873571963933045e-03
            5.00000000000000000000000000000000e-01 4.87668880442191920998595833225409e-01 1.23311195578080790014041667745914e-02 2.48800419844458437293455155270294e-03
            4.10757909252076280903764882168616e-01 5.74710061408912786795610827539349e-01 1.45320293390108767894730590342078e-02 2.88495418493539936347080931966502e-03
            3.24384118273061905490806111629354e-01 6.58953681301480465037911926629022e-01 1.66622004254576294712819617416244e-02 3.14750131495105528456335264309018e-03
            2.43654731456761419750023378583137e-01 7.37692100676460293939840084931348e-01 1.86531678667782863101365364855155e-02 3.23084431280764965144225442372772e-03
            1.71164420391654581443674487672979e-01 8.08394638356514039223554846103070e-01 2.04409412518313793327706662239507e-02 3.10589531739593333048432555187901e-03
            1.09242998051599260467270369190373e-01 8.68788939772439983144636244105641e-01 2.19680621759608118992446179618128e-02 2.76358496010473131399010071618250e-03
            5.98804231365069883352703072887380e-02 9.16934123061613703953298681881279e-01 2.31854538018793077114310108299833e-02 2.21683240587659206172199688467117e-03
            2.46622391156161580028083335491829e-02 9.51283747806963564386251164251007e-01 2.40540130774202776109405021998100e-02 1.49995214231954496382492880712789e-03
            4.71226234279131794835393520770594e-03 9.70741713482266010970533898216672e-01 2.45460241749426710811121665756218e-02 6.66471299188430726413712168465509e-04
            9.95287737657208571029343602276640e-01 4.43009007977485016332996892174378e-03 2.82172263016578807326428801616203e-04 4.83826509084280636759697186155549e-06
            9.75337760884383841997191666450817e-01 2.31854538018793979170517616239522e-02 1.47678531373676008575657192523067e-03 5.81543584650973195064420362410118e-05
            9.40119576863492900642427230195608e-01 5.62947580614999651404950498090329e-02 3.58566507500713421707771999535908e-03 2.16502056708442475055700904285061e-04
            8.90757001948400684021578399551800e-01 1.02701481103568928454627950941358e-01 6.54151694803038752379364950684248e-03 5.19678305180139494569291347403350e-04
            8.28835579608345418556325512327021e-01 1.60915022472687324617268700421846e-01 1.02493979189672568264057872511330e-02 9.83465937782175688916530553740358e-04
            7.56345268543238580249976621416863e-01 2.29064583037918545471711695427075e-01 1.45901484188428742783116831560619e-02 1.59587248333772804657904043068584e-03
            6.75615881726938205531496350886300e-01 3.04959860012108063820335246418836e-01 1.94242582609537306481684026948642e-02 2.31714416114221300260767044676413e-03
            5.89242090747923774607386349089211e-01 3.86161551839394878715694403581438e-01 2.45963574126813466769192473293515e-02 3.08360324742398437744550498962326e-03
            5.00000000000000000000000000000000e-01 4.70059788431746450321213615097804e-01 2.99402115682535496787863849021960e-02 3.81485916705116357219851330739857e-03
            4.10757909252076280903764882168616e-01 5.53958025024097855393279132840689e-01 3.52840657238258081918047537328675e-02 4.42350295301101661959997102258058e-03
            3.24384118273061905490806111629354e-01 6.35159716851384725799789521261118e-01 4.04561648755533687094043671095278e-02 4.82606671329299449779703934382269e-03
            2.43654731456761419750023378583137e-01 7.11054993825574355170715534768533e-01 4.52902747176642250792610866483301e-02 4.95385660994249361899210981619035e-03
            1.71164420391654581443674487672979e-01 7.79204554390805603780734145402675e-01 4.96310252175398147755913669243455e-02 4.76227219828506426885228464129796e-03
            1.09242998051599260467270369190373e-01 8.37418095759924097087889549584361e-01 5.33389061884766979559913124830928e-02 4.23740740693733138055865339310913e-03
            5.98804231365069883352703072887380e-02 8.83824818801993039585340738995001e-01 5.62947580614999720793889537162613e-02 3.39907120360221758764707189470755e-03
            2.46622391156161580028083335491829e-02 9.16934123061613592930996219365625e-01 5.84036378227702490661954470851924e-02 2.29987802425856606783338698107855e-03
            4.71226234279131794835393520770594e-03 9.35689486783718105122886754543288e-01 5.95982508734905769287593102490064e-02 1.02190106707816829624546617338865e-03
            9.95287737657208571029343602276640e-01 4.19748067685924049208257002874234e-03 5.14781665932188478573827694617648e-04 6.36580704579817623127652517767139e-06
            9.75337760884383841997191666450817e-01 2.19680621759608743492897531268682e-02 2.69417693965528365351858042231470e-03 7.65149114217927395296264814206211e-05
            9.40119576863492900642427230195608e-01 5.33389061884767118337791202975495e-02 6.54151694803038752379364950684248e-03 2.84856305338226218053926386275521e-04
            8.90757001948400684021578399551800e-01 9.73089654282975891064211282355245e-02 1.19340326233017268720004722126760e-02 6.83751666051831872260302613852900e-04
            8.28835579608345418556325512327021e-01 1.52465905948305946449039538492798e-01 1.86985144433486349946349491801811e-02 1.29396679976989865436254323327603e-03
            7.56345268543238580249976621416863e-01 2.17037158102967475681310816071345e-01 2.66175733537939440687125625117915e-02 2.09972296016898906864667218030718e-03
            6.75615881726938205531496350886300e-01 2.88947424672587938054846290469868e-01 3.54366936004738564136573586438317e-02 3.04871526263554223912244367511448e-03
            5.89242090747923774607386349089211e-01 3.65885483771972608302291973814135e-01 4.48724254801036170903216770966537e-02 4.05716158795224948740987258588575e-03
            5.00000000000000000000000000000000e-01 4.45378500974200342010789199775900e-01 5.46214990257996579892108002241002e-02 5.01929036718237481934368204861130e-03
            4.10757909252076280903764882168616e-01 5.24871518176427964696983963222010e-01 6.43705725714956988880999233515467e-02 5.82009578571507195376977250589334e-03
            3.24384118273061905490806111629354e-01 6.01809577275812634944429646566277e-01 7.38063044511254595647642418043688e-02 6.34975738413311521779469615012204e-03
            2.43654731456761419750023378583137e-01 6.73719843845433152829116352222627e-01 8.26254246978054274208602691942360e-02 6.51789323638578711045399316503790e-03
            1.71164420391654581443674487672979e-01 7.38291096000094793083690092316829e-01 9.05444836082506254726354200101923e-02 6.26582160023211008625931839333134e-03
            1.09242998051599260467270369190373e-01 7.93448036520103205937459733831929e-01 9.73089654282975891064211282355245e-02 5.57524596534667883640512542342549e-03
            5.98804231365069883352703072887380e-02 8.37418095759924097087889549584361e-01 1.02701481103568914576840143126901e-01 4.47222940677925034913542390313523e-03
            2.46622391156161580028083335491829e-02 8.68788939772439872122333781589987e-01 1.06548821111943969874857884860830e-01 3.02599784352684938668054392962858e-03
            4.71226234279131794835393520770594e-03 8.86559521271541584042097383644432e-01 1.08728216385667098009548681147862e-01 1.34453670701654222525978088498277e-03
            9.95287737657208571029343602276640e-01 3.90569069015411383591351857091922e-03 8.06571652637315134742879152440764e-04 7.68879572985607591699072638213863e-06
            9.75337760884383841997191666450817e-01 2.04409412518313723938767623167223e-02 4.22129786378478560893157123246056e-03 9.24168011970320492321981364902683e-05
            9.40119576863492900642427230195608e-01 4.96310252175398355922730786460306e-02 1.02493979189672637652996911583614e-02 3.44057230819272213354770295623553e-04
            8.90757001948400684021578399551800e-01 9.05444836082506671059988434535626e-02 1.86985144433486488724227569946379e-02 8.25853949451923097409278273772770e-04
            8.28835579608345418556325512327021e-01 1.41867161583643514033425958587031e-01 2.92972588080110674102485290859477e-02 1.56288846536957351344987454666580e-03
            7.56345268543238580249976621416863e-01 2.01949710571280610649225195629697e-01 4.17050208854808091007981829534401e-02 2.53610277752360424238786684725255e-03
            6.75615881726938205531496350886300e-01 2.68861098684595234420413589759846e-01 5.55230195884665600480900593538536e-02 3.68232161676525747429078450068118e-03
            5.89242090747923774607386349089211e-01 3.40450769793656682349336506376858e-01 7.03071394584195430432771445339313e-02 4.90035064970650770937998785825584e-03
            5.00000000000000000000000000000000e-01 4.14417789804172709278162756163510e-01 8.55822101958272907218372438364895e-02 6.06243608460813650923304507500688e-03
            4.10757909252076280903764882168616e-01 4.88384809814688569673535312176682e-01 1.00857280933235093911548574396875e-01 7.02967075543026870265705952078861e-03
            3.24384118273061905490806111629354e-01 5.59974480923750128624760691309348e-01 1.15641400803187965884433197061298e-01 7.66941050985362436642756023275069e-03
            2.43654731456761419750023378583137e-01 6.26885869037064780151524701068411e-01 1.29459399506173800098451920348452e-01 7.87248958742154357481890514236511e-03
            1.71164420391654581443674487672979e-01 6.86968418024701876767323938111076e-01 1.41867161583643541789001574215945e-01 7.56803057606091761355671110322874e-03
            1.09242998051599260467270369190373e-01 7.38291096000094793083690092316829e-01 1.52465905948306001960190769750625e-01 6.73393445055009507577148397672318e-03
            5.98804231365069883352703072887380e-02 7.79204554390805714803036607918330e-01 1.60915022472687296861693084792932e-01 5.40168090524798232693326482944940e-03
            2.46622391156161580028083335491829e-02 8.08394638356514039223554846103070e-01 1.66943122527869802773636820347747e-01 3.65488289709002545091465208315640e-03
            4.71226234279131794835393520770594e-03 8.24929888918191411839586635323940e-01 1.70357848739017270212059429468354e-01 1.62396818143697410814774162446383e-03
            9.95287737657208571029343602276640e-01 3.56409732710477383452185584644667e-03 1.14816501568665513613454187691332e-03 8.76467529350308184772633257608732e-06
            9.75337760884383841997191666450817e-01 1.86531678667783071268182482072007e-02 6.00907124883785087599008534198219e-03 1.05348520446566060355941818471592e-04
            9.40119576863492900642427230195608e-01 4.52902747176642181403671827411017e-02 1.45901484188428812172055870632903e-02 3.92200549535110225968237429583496e-04
            8.90757001948400684021578399551800e-01 8.26254246978053719097090379364090e-02 2.66175733537939440687125625117915e-02 9.41414229369662920905714731389935e-04
            8.28835579608345418556325512327021e-01 1.29459399506173772342876304719539e-01 4.17050208854808091007981829534401e-02 1.78158067923883558884845523806462e-03
            7.56345268543238580249976621416863e-01 1.84287103295494886223693242754962e-01 5.93676281612665335263301358281751e-02 2.89097514577380454089960970520679e-03
            6.75615881726938205531496350886300e-01 2.45346393046400601534173802065197e-01 7.90377252266611929343298470485024e-02 4.19758235634631708677044770183784e-03
            5.89242090747923774607386349089211e-01 3.10674801179520787552945648712921e-01 1.00083108072555437839668002197868e-01 5.58604803379115881500904805534446e-03
            5.00000000000000000000000000000000e-01 3.78172634271619290124988310708432e-01 1.21827365728380709875011689291568e-01 6.91074202464227876907543191009609e-03
            4.10757909252076280903764882168616e-01 4.45670467363717681674728510188288e-01 1.43571623384205981910355376385269e-01 8.01332012922836671442450295899107e-03
            3.24384118273061905490806111629354e-01 5.10998875496837867693500356836012e-01 1.64617006230100226815693531534635e-01 8.74257753401197316989179597612747e-03
            2.43654731456761419750023378583137e-01 5.72058165247743666270707763032988e-01 1.84287103295494913979268858383875e-01 8.97407310709312511232127462790231e-03
            1.71164420391654581443674487672979e-01 6.26885869037064780151524701068411e-01 2.01949710571280638404800811258610e-01 8.62701168570629249587877751537235e-03
            1.09242998051599260467270369190373e-01 6.73719843845433263851418814738281e-01 2.17037158102967531192462047329172e-01 7.67620196718510686140213294947898e-03
            5.98804231365069883352703072887380e-02 7.11054993825574466193017997284187e-01 2.29064583037918545471711695427075e-01 6.15752854374511555957116826220954e-03
            2.46622391156161580028083335491829e-02 7.37692100676460293939840084931348e-01 2.37645660207923548057351581519470e-01 4.16630418524222348453101005816279e-03
            4.71226234279131794835393520770594e-03 7.52781171216133859758201651857235e-01 2.42506566441074822293444412935060e-01 1.85120717175590865366530124447308e-03
            9.95287737657208571029343602276640e-01 3.18367927765367871723034554065634e-03 1.52858306513775025342605218270364e-03 9.55885701160958090830555705652571e-06
            9.75337760884383841997191666450817e-01 1.66622004254576329407289136952386e-02 8.00003869015852506207941985394427e-03 1.14894324046415339819066092630351e-04
            9.40119576863492900642427230195608e-01 4.04561648755533964649799827384413e-02 1.94242582609537028925927870659507e-02 4.27738489714479407002162281159485e-04
            8.90757001948400684021578399551800e-01 7.38063044511254595647642418043688e-02 3.54366936004738564136573586438317e-02 1.02671732903895853057651965656305e-03
            8.28835579608345418556325512327021e-01 1.15641400803188035273372236133582e-01 5.55230195884665461703022515393968e-02 1.94301264989404681889084702817172e-03
            7.56345268543238580249976621416863e-01 1.64617006230100226815693531534635e-01 7.90377252266611929343298470485024e-02 3.15293118309280685079842854179333e-03
            6.75615881726938205531496350886300e-01 2.19159062085270056607910760249069e-01 1.05225056187791737860592888864630e-01 4.57793223309848305857538974805720e-03
            5.89242090747923774607386349089211e-01 2.77514567035655135285310279869009e-01 1.33243342216421090107303371041780e-01 6.09220908098821734011352191373589e-03
            5.00000000000000000000000000000000e-01 3.37807940863469102765748175443150e-01 1.62192059136530897234251824556850e-01 7.53693578433461269794335635197058e-03
            4.10757909252076280903764882168616e-01 3.98101314691282959223883608501637e-01 1.91140776056640704361200278071919e-01 8.73942031375949356097887488203924e-03
            3.24384118273061905490806111629354e-01 4.56456819641668065656858743750490e-01 2.19159062085270028852335144620156e-01 9.53475693759896734957859365522381e-03
            2.43654731456761419750023378583137e-01 5.10998875496837978715802819351666e-01 2.45346393046400601534173802065197e-01 9.78722870726550855935421679987485e-03
            1.71164420391654581443674487672979e-01 5.59974480923750128624760691309348e-01 2.68861098684595289931564821017673e-01 9.40871947672483437130619421395750e-03
            1.09242998051599260467270369190373e-01 6.01809577275812856989034571597585e-01 2.88947424672587938054846290469868e-01 8.37175531772970868416017253821337e-03
            5.98804231365069883352703072887380e-02 6.35159716851384836822091983776772e-01 3.04959860012108174842637708934490e-01 6.71547238472074495274632965902128e-03
            2.46622391156161580028083335491829e-02 6.58953681301480576060214389144676e-01 3.16384079582903265936977277306141e-01 4.54381989520157188178206553175187e-03
            4.71226234279131794835393520770594e-03 6.72432202449284588396949402522296e-01 3.22855535207924093654696662269998e-01 2.01894811400461738834510860840510e-03
            9.95287737657208571029343602276640e-01 2.77666331501913096757583332419017e-03 1.93559902777229800308056439916982e-03 1.00458121707907386269608904227901e-05
            9.75337760884383841997191666450817e-01 1.45320293390109236270069104079994e-02 1.01302097766052343758014231411835e-02 1.20747365240261240646423013256339e-04
            9.40119576863492900642427230195608e-01 3.52840657238257873751230420111824e-02 2.45963574126813119824497277932096e-02 4.49528695812744807057403928496342e-04
            8.90757001948400684021578399551800e-01 6.43705725714957405214633467949170e-02 4.48724254801035754569582536532835e-02 1.07902120802668604129803675562016e-03
            8.28835579608345418556325512327021e-01 1.00857280933235107789336382211332e-01 7.03071394584194736543381054616475e-02 2.04199519907027286086398376596662e-03
            7.56345268543238580249976621416863e-01 1.43571623384206009665930992014182e-01 1.00083108072555410084092386568955e-01 3.31355039774215969633486267298395e-03
            6.75615881726938205531496350886300e-01 1.91140776056640704361200278071919e-01 1.33243342216421090107303371041780e-01 4.81114502376807102318467812551717e-03
            5.89242090747923774607386349089211e-01 2.42035849238939310135165783322009e-01 1.68722060013136915257447867588780e-01 6.40256340883248002693317602052048e-03
            5.00000000000000000000000000000000e-01 2.94621045373961887303693174544605e-01 2.05378954626038112696306825455395e-01 7.92088856866241322607358199547889e-03
            4.10757909252076280903764882168616e-01 3.47206241508984381205493718880462e-01 2.42035849238939282379590167693095e-01 9.18463105442326663285790289137367e-03
            3.24384118273061905490806111629354e-01 3.98101314691283014735034839759464e-01 2.77514567035655079774159048611182e-01 1.00204843709796580925042164267325e-02
            2.43654731456761419750023378583137e-01 4.45670467363717737185879741446115e-01 3.10674801179520843064096879970748e-01 1.02858177652774083338238497731254e-02
            1.71164420391654581443674487672979e-01 4.88384809814688680695837774692336e-01 3.40450769793656737860487737634685e-01 9.88802620606651569423739545072749e-03
            1.09242998051599260467270369190373e-01 5.24871518176428075719286425737664e-01 3.65885483771972719324594436329789e-01 8.79823616564065838496766502885293e-03
            5.98804231365069883352703072887380e-02 5.53958025024098077437884057871997e-01 3.86161551839394934226845634839265e-01 7.05757750462228383714347046407056e-03
            2.46622391156161580028083335491829e-02 5.74710061408912897817913290055003e-01 4.00627699475470944179278376395814e-01 4.77529490708542610905018577227565e-03
            4.71226234279131794835393520770594e-03 5.86465427432904751192666026327061e-01 4.08822310224303930858980038465234e-01 2.12179903007539459938723958032369e-03
            9.95287737657208571029343602276640e-01 2.35613117139571448532819886167999e-03 2.35613117139571448532819886167999e-03 1.02098885048394470053323077052276e-05
            9.75337760884383841997191666450817e-01 1.23311195578081067569797824035049e-02 1.23311195578080512458285511456779e-02 1.22719508925394775499320698664008e-04
            9.40119576863492900642427230195608e-01 2.99402115682535496787863849021960e-02 2.99402115682535496787863849021960e-02 4.56870762258413407128770922227545e-04
            8.90757001948400684021578399551800e-01 5.46214990257996579892108002241002e-02 5.46214990257996579892108002241002e-02 1.09664465560503068472519849763103e-03
            8.28835579608345418556325512327021e-01 8.55822101958272907218372438364895e-02 8.55822101958272907218372438364895e-02 2.07534671716680732544979548492847e-03
            7.56345268543238580249976621416863e-01 1.21827365728380709875011689291568e-01 1.21827365728380709875011689291568e-01 3.36766998436234012148648986340049e-03
            6.75615881726938205531496350886300e-01 1.62192059136530897234251824556850e-01 1.62192059136530897234251824556850e-01 4.88972453776414118370441386218772e-03
            5.89242090747923774607386349089211e-01 2.05378954626038084940731209826481e-01 2.05378954626038140451882441084308e-01 6.50713525576488652918394350876952e-03
            5.00000000000000000000000000000000e-01 2.50000000000000000000000000000000e-01 2.50000000000000000000000000000000e-01 8.05025893082519233834304372976476e-03
            4.10757909252076280903764882168616e-01 2.94621045373961831792541943286778e-01 2.94621045373961831792541943286778e-01 9.33464188155991130002586686487120e-03
            3.24384118273061905490806111629354e-01 3.37807940863469047254596944185323e-01 3.37807940863469047254596944185323e-01 1.01841470309050729364797049925073e-02
            2.43654731456761419750023378583137e-01 3.78172634271619290124988310708432e-01 3.78172634271619290124988310708432e-01 1.04538140649222347638991337248626e-02
            1.71164420391654581443674487672979e-01 4.14417789804172709278162756163510e-01 4.14417789804172709278162756163510e-01 1.00495254520494570193989147810498e-02
            1.09242998051599260467270369190373e-01 4.45378500974200397521940431033727e-01 4.45378500974200397521940431033727e-01 8.94193607875972372445172453581108e-03
            5.98804231365069883352703072887380e-02 4.70059788431746505832364846355631e-01 4.70059788431746505832364846355631e-01 7.17284757184392766926617213130157e-03
            2.46622391156161580028083335491829e-02 4.87668880442191920998595833225409e-01 4.87668880442191920998595833225409e-01 4.85328888796376575753832582904579e-03
            4.71226234279131794835393520770594e-03 4.97643868828604341025823032396147e-01 4.97643868828604341025823032396147e-01 2.15645396892196394497687172986389e-03
            9.95287737657208571029343602276640e-01 1.93559902777229821992099889627070e-03 2.77666331501913096757583332419017e-03 1.00458121707907098278406837765786e-05
            9.75337760884383841997191666450817e-01 1.01302097766052898869526543990105e-02 1.45320293390108681158556791501724e-02 1.20747365240260888280716955467398e-04
            9.40119576863492900642427230195608e-01 2.45963574126813119824497277932096e-02 3.52840657238257873751230420111824e-02 4.49528695812743451804688321615799e-04
            8.90757001948400684021578399551800e-01 4.48724254801035754569582536532835e-02 6.43705725714957405214633467949170e-02 1.07902120802668278869151929910686e-03
            8.28835579608345418556325512327021e-01 7.03071394584194875321259132761043e-02 1.00857280933235093911548574396875e-01 2.04199519907026678933181784714179e-03
            7.56345268543238580249976621416863e-01 1.00083108072555423961880194383411e-01 1.43571623384205981910355376385269e-01 3.31355039774214972167487580634315e-03
            6.75615881726938205531496350886300e-01 1.33243342216421090107303371041780e-01 1.91140776056640704361200278071919e-01 4.81114502376805627803513232265686e-03
            5.89242090747923774607386349089211e-01 1.68722060013136887501872251959867e-01 2.42035849238939337890741398950922e-01 6.40256340883246094497494027564244e-03
            5.00000000000000000000000000000000e-01 2.05378954626038140451882441084308e-01 2.94621045373961831792541943286778e-01 7.92088856866239067466839429698666e-03
            4.10757909252076280903764882168616e-01 2.42035849238939337890741398950922e-01 3.47206241508984325694342487622635e-01 9.18463105442323887728228726246016e-03
            3.24384118273061905490806111629354e-01 2.77514567035655135285310279869009e-01 3.98101314691282959223883608501637e-01 1.00204843709796286022051248210119e-02
            2.43654731456761419750023378583137e-01 3.10674801179520843064096879970748e-01 4.45670467363717737185879741446115e-01 1.02858177652773788435247581674048e-02
            1.71164420391654581443674487672979e-01 3.40450769793656793371638968892512e-01 4.88384809814688625184686543434509e-01 9.88802620606648620393830384500689e-03
            1.09242998051599260467270369190373e-01 3.65885483771972774835745667587616e-01 5.24871518176428075719286425737664e-01 8.79823616564063236411552537674652e-03
            5.98804231365069883352703072887380e-02 3.86161551839394989737996866097092e-01 5.53958025024098077437884057871997e-01 7.05757750462226302046175874238543e-03
            2.46622391156161580028083335491829e-02 4.00627699475470999690429607653641e-01 5.74710061408912897817913290055003e-01 4.77529490708541223126237795781890e-03
            4.71226234279131794835393520770594e-03 4.08822310224304041881282500980888e-01 5.86465427432904640170363563811406e-01 2.12179903007538809417420466729709e-03
            9.95287737657208571029343602276640e-01 1.52858306513775068710692117690542e-03 3.18367927765367828354947654645457e-03 9.55885701160957413204197902212300e-06
            9.75337760884383841997191666450817e-01 8.00003869015858404267760306538548e-03 1.66622004254575739601307304837974e-02 1.14894324046415244951376000148713e-04
            9.40119576863492900642427230195608e-01 1.94242582609537098314866909731791e-02 4.04561648755533895260860788312129e-02 4.27738489714479027531401911232933e-04
            8.90757001948400684021578399551800e-01 3.54366936004738633525512625510601e-02 7.38063044511254595647642418043688e-02 1.02671732903895766321478166815950e-03
            8.28835579608345418556325512327021e-01 5.55230195884665669869839632610820e-02 1.15641400803188021395584428319125e-01 1.94301264989404508416737105136463e-03
            7.56345268543238580249976621416863e-01 7.90377252266612345676932704918727e-02 1.64617006230100171304542300276808e-01 3.15293118309280424871321457658269e-03
            6.75615881726938205531496350886300e-01 1.05225056187791779493956312308001e-01 2.19159062085270028852335144620156e-01 4.57793223309847958912843779444302e-03
            5.89242090747923774607386349089211e-01 1.33243342216421117862878986670694e-01 2.77514567035655135285310279869009e-01 6.09220908098821213594309398331461e-03
            5.00000000000000000000000000000000e-01 1.62192059136530952745403055814677e-01 3.37807940863469047254596944185323e-01 7.53693578433460662641119043314575e-03
            4.10757909252076280903764882168616e-01 1.91140776056640759872351509329746e-01 3.98101314691282903712732377243810e-01 8.73942031375948662208497097481086e-03
            3.24384118273061905490806111629354e-01 2.19159062085270084363486375877983e-01 4.56456819641668010145707512492663e-01 9.53475693759895867596121377118834e-03
            2.43654731456761419750023378583137e-01 2.45346393046400684800900648951938e-01 5.10998875496837867693500356836012e-01 9.78722870726549988573683691583938e-03
            1.71164420391654581443674487672979e-01 2.68861098684595345442716052275500e-01 5.59974480923750128624760691309348e-01 9.40871947672482569768881432992202e-03
            1.09242998051599260467270369190373e-01 2.88947424672588104588299984243349e-01 6.01809577275812745966732109081931e-01 8.37175531772970347998974460779209e-03
            5.98804231365069883352703072887380e-02 3.04959860012108230353788940192317e-01 6.35159716851384725799789521261118e-01 6.71547238472073974857590172860000e-03
            2.46622391156161580028083335491829e-02 3.16384079582903321448128508563968e-01 6.58953681301480465037911926629022e-01 4.54381989520156841233511357813768e-03
            4.71226234279131794835393520770594e-03 3.22855535207924204676999124785652e-01 6.72432202449284477374646940006642e-01 2.01894811400461565362163263159800e-03
            9.95287737657208571029343602276640e-01 1.14816501568665513613454187691332e-03 3.56409732710477383452185584644667e-03 8.76467529350309540025348864489274e-06
            9.75337760884383841997191666450817e-01 6.00907124883790551977957861140567e-03 1.86531678667782516156670169493736e-02 1.05348520446566236538794847366063e-04
            9.40119576863492900642427230195608e-01 1.45901484188428812172055870632903e-02 4.52902747176642181403671827411017e-02 3.92200549535110822279432296610935e-04
            8.90757001948400684021578399551800e-01 2.66175733537939440687125625117915e-02 8.26254246978053719097090379364090e-02 9.41414229369664438788756211096143e-04
            8.28835579608345418556325512327021e-01 4.17050208854808160396920868606685e-02 1.29459399506173772342876304719539e-01 1.78158067923883840777410370037614e-03
            7.56345268543238580249976621416863e-01 5.93676281612665196485423280137184e-02 1.84287103295494913979268858383875e-01 2.89097514577380887770829964722452e-03
            6.75615881726938205531496350886300e-01 7.90377252266612068121176548629592e-02 2.45346393046400601534173802065197e-01 4.19758235634632402566435160906622e-03
            5.89242090747923774607386349089211e-01 1.00083108072555396206304578754498e-01 3.10674801179520843064096879970748e-01 5.58604803379116835598816592778348e-03
            5.00000000000000000000000000000000e-01 1.21827365728380709875011689291568e-01 3.78172634271619290124988310708432e-01 6.91074202464229004477802575934220e-03
            4.10757909252076280903764882168616e-01 1.43571623384205981910355376385269e-01 4.45670467363717681674728510188288e-01 8.01332012922838059221231077344783e-03
            3.24384118273061905490806111629354e-01 1.64617006230100199060117915905721e-01 5.10998875496837867693500356836012e-01 8.74257753401198878240307976739132e-03
            2.43654731456761419750023378583137e-01 1.84287103295494886223693242754962e-01 5.72058165247743666270707763032988e-01 8.97407310709313899010908244235907e-03
            1.71164420391654581443674487672979e-01 2.01949710571280610649225195629697e-01 6.26885869037064780151524701068411e-01 8.62701168570630637366658532982910e-03
            1.09242998051599260467270369190373e-01 2.17037158102967503436886431700259e-01 6.73719843845433263851418814738281e-01 7.67620196718511900446646478712864e-03
            5.98804231365069883352703072887380e-02 2.29064583037918573227287311055989e-01 7.11054993825574466193017997284187e-01 6.15752854374512510055028613464856e-03
            2.46622391156161580028083335491829e-02 2.37645660207923520301775965890556e-01 7.37692100676460293939840084931348e-01 4.16630418524223042342491396539117e-03
            4.71226234279131794835393520770594e-03 2.42506566441074794537868797306146e-01 7.52781171216133859758201651857235e-01 1.85120717175591168943138420388550e-03
            9.95287737657208571029343602276640e-01 8.06571652637315243163096400991208e-04 3.90569069015411383591351857091922e-03 7.68879572985606914072714834773592e-06
            9.75337760884383841997191666450817e-01 4.22129786378484112008280249028758e-03 2.04409412518313168827255310588953e-02 9.24168011970319543645080440086303e-05
            9.40119576863492900642427230195608e-01 1.02493979189672602958527392047472e-02 4.96310252175398425311669825532590e-02 3.44057230819271888094118549972222e-04
            8.90757001948400684021578399551800e-01 1.86985144433486488724227569946379e-02 9.05444836082506671059988434535626e-02 8.25853949451922338467757533919666e-04
            8.28835579608345418556325512327021e-01 2.92972588080110570019076732251051e-02 1.41867161583643514033425958587031e-01 1.56288846536957199556683306695959e-03
            7.56345268543238580249976621416863e-01 4.17050208854808160396920868606685e-02 2.01949710571280610649225195629697e-01 2.53610277752360164030265288204191e-03
            6.75615881726938205531496350886300e-01 5.55230195884665531091961554466252e-02 2.68861098684595234420413589759846e-01 3.68232161676525357116296355286522e-03
            5.89242090747923774607386349089211e-01 7.03071394584194736543381054616475e-02 3.40450769793656737860487737634685e-01 4.90035064970650250520955992783456e-03
            5.00000000000000000000000000000000e-01 8.55822101958272907218372438364895e-02 4.14417789804172709278162756163510e-01 6.06243608460813043770087915618205e-03
            4.10757909252076280903764882168616e-01 1.00857280933235080033760766582418e-01 4.88384809814688569673535312176682e-01 7.02967075543026176376315561356023e-03
            3.24384118273061905490806111629354e-01 1.15641400803188007517796620504669e-01 5.59974480923750128624760691309348e-01 7.66941050985361656017191833711877e-03
            2.43654731456761419750023378583137e-01 1.29459399506173772342876304719539e-01 6.26885869037064780151524701068411e-01 7.87248958742153663592500123513673e-03
            1.71164420391654581443674487672979e-01 1.41867161583643514033425958587031e-01 6.86968418024701876767323938111076e-01 7.56803057606090980730106920759681e-03
            1.09242998051599260467270369190373e-01 1.52465905948305946449039538492798e-01 7.38291096000094793083690092316829e-01 6.73393445055008813687758006949480e-03
            5.98804231365069883352703072887380e-02 1.60915022472687352372844316050760e-01 7.79204554390805714803036607918330e-01 5.40168090524797712276283689902812e-03
            2.46622391156161580028083335491829e-02 1.66943122527869747262485589089920e-01 8.08394638356514150245857308618724e-01 3.65488289709002198146770012954221e-03
            4.71226234279131794835393520770594e-03 1.70357848739017297967635045097268e-01 8.24929888918191411839586635323940e-01 1.62396818143697237342426564765674e-03
            9.95287737657208571029343602276640e-01 5.14781665932188695414262191718535e-04 4.19748067685924049208257002874234e-03 6.36580704579817877237536694057241e-06
            9.75337760884383841997191666450817e-01 2.69417693965533699626546670913285e-03 2.19680621759608223075854738226553e-02 7.65149114217927666346807935582319e-05
            9.40119576863492900642427230195608e-01 6.54151694803038405434669755322830e-03 5.33389061884767118337791202975495e-02 2.84856305338226326474143634825964e-04
            8.90757001948400684021578399551800e-01 1.19340326233017251372769962358689e-02 9.73089654282975891064211282355245e-02 6.83751666051832089100737110953787e-04
            8.28835579608345418556325512327021e-01 1.86985144433486384640819011337953e-02 1.52465905948305946449039538492798e-01 1.29396679976989908804341222747780e-03
            7.56345268543238580249976621416863e-01 2.66175733537939301909247546973347e-02 2.17037158102967475681310816071345e-01 2.09972296016898993600841016871072e-03
            6.75615881726938205531496350886300e-01 3.54366936004738355969756469221466e-02 2.88947424672587938054846290469868e-01 3.04871526263554354016505065771980e-03
            5.89242090747923774607386349089211e-01 4.48724254801035407624887341171416e-02 3.65885483771972663813443205071962e-01 4.05716158795225122213334856269285e-03
            5.00000000000000000000000000000000e-01 5.46214990257996302336351845951867e-02 4.45378500974200397521940431033727e-01 5.01929036718237742142889601382194e-03
            4.10757909252076280903764882168616e-01 6.43705725714956988880999233515467e-02 5.24871518176427964696983963222010e-01 5.82009578571507368849324848270044e-03
            3.24384118273061905490806111629354e-01 7.38063044511254179314008183609985e-02 6.01809577275812634944429646566277e-01 6.34975738413311695251817212692913e-03
            2.43654731456761419750023378583137e-01 8.26254246978053302763456144930387e-02 6.73719843845433263851418814738281e-01 6.51789323638578884517746914184499e-03
            1.71164420391654581443674487672979e-01 9.05444836082506254726354200101923e-02 7.38291096000094793083690092316829e-01 6.26582160023211268834453235854198e-03
            1.09242998051599260467270369190373e-01 9.73089654282975474730577047921543e-02 7.93448036520103205937459733831929e-01 5.57524596534668057112860140023258e-03
            5.98804231365069883352703072887380e-02 1.02701481103568886821264527497988e-01 8.37418095759924097087889549584361e-01 4.47222940677925208385889987994233e-03
            2.46622391156161580028083335491829e-02 1.06548821111943928241494461417460e-01 8.68788939772439872122333781589987e-01 3.02599784352685025404228191803213e-03
            4.71226234279131794835393520770594e-03 1.08728216385667084131760873333405e-01 8.86559521271541584042097383644432e-01 1.34453670701654265894064987918455e-03
            9.95287737657208571029343602276640e-01 2.82172263016578482065777055964872e-04 4.43009007977485016332996892174378e-03 4.83826509084281568495939165885922e-06
            9.75337760884383841997191666450817e-01 1.47678531373681277798215472074617e-03 2.31854538018793458753474823197394e-02 5.81543584650974279266592847914552e-05
            9.40119576863492900642427230195608e-01 3.58566507500712814554555407653424e-03 5.62947580614999720793889537162613e-02 2.16502056708442854526461274211613e-04
            8.90757001948400684021578399551800e-01 6.54151694803037538072931766919282e-03 1.02701481103568942332415758755815e-01 5.19678305180140470351246584357341e-04
            8.28835579608345418556325512327021e-01 1.02493979189672412138945034598692e-02 1.60915022472687352372844316050760e-01 9.83465937782177423640006530547453e-04
            7.56345268543238580249976621416863e-01 1.45901484188428534616299714343768e-02 2.29064583037918573227287311055989e-01 1.59587248333773086550468889299736e-03
            6.75615881726938205531496350886300e-01 1.94242582609536681981232675298088e-02 3.04959860012108119331486477676663e-01 2.31714416114221733941636038878187e-03
            5.89242090747923774607386349089211e-01 2.45963574126812599407454484889968e-02 3.86161551839394989737996866097092e-01 3.08360324742399001529680191424632e-03
            5.00000000000000000000000000000000e-01 2.99402115682534941676351536443690e-02 4.70059788431746505832364846355631e-01 3.81485916705117051109241721462695e-03
            4.10757909252076280903764882168616e-01 3.52840657238257179861840029388986e-02 5.53958025024097966415581595356343e-01 4.42350295301102442585561291821250e-03
            3.24384118273061905490806111629354e-01 4.04561648755533131982531358517008e-02 6.35159716851384725799789521261118e-01 4.82606671329300317141441922785816e-03
            2.43654731456761419750023378583137e-01 4.52902747176641348736403358543612e-02 7.11054993825574466193017997284187e-01 4.95385660994250315997122768862937e-03
            1.71164420391654581443674487672979e-01 4.96310252175397453866523278520617e-02 7.79204554390805714803036607918330e-01 4.76227219828507294246966452533343e-03
            1.09242998051599260467270369190373e-01 5.33389061884766216281583695035806e-02 8.37418095759924208110192012100015e-01 4.23740740693733831945255730033750e-03
            5.98804231365069883352703072887380e-02 5.62947580614998679959803951078356e-02 8.83824818801993150607643201510655e-01 3.39907120360222365917923781353238e-03
            2.46622391156161580028083335491829e-02 5.84036378227701796772564080129087e-02 9.16934123061613703953298681881279e-01 2.29987802425857040464207692309628e-03
            4.71226234279131794835393520770594e-03 5.95982508734904173341995203827537e-02 9.35689486783718216145189217058942e-01 1.02190106707817024780937664729663e-03
            9.95287737657208571029343602276640e-01 1.16214940673435815116261438095790e-04 4.59604740211799322330277206560822e-03 3.15545694666093726719630768728919e-06
            9.75337760884383841997191666450817e-01 6.08226038195829000684855536462692e-04 2.40540130774203296526447815040228e-02 3.79275568725268082018002580024785e-05
            9.40119576863492900642427230195608e-01 1.47678531373681212746085122944351e-03 5.84036378227702906995588705285627e-02 1.41199976847079030229520357764272e-04
            8.90757001948400684021578399551800e-01 2.69417693965533222577590777291334e-03 1.06548821111943983752645692675287e-01 3.38927794843916477176420931627376e-04
            8.28835579608345418556325512327021e-01 4.22129786378483157910368461784856e-03 1.66943122527869747262485589089920e-01 6.41404380891126297409288170570107e-04
            7.56345268543238580249976621416863e-01 6.00907124883789250935350878535246e-03 2.37645660207923520301775965890556e-01 1.04080839288115041020577056229968e-03
            6.75615881726938205531496350886300e-01 8.00003869015856322599589134370035e-03 3.16384079582903210425826046048314e-01 1.51121290429693383439158083092480e-03
            5.89242090747923774607386349089211e-01 1.01302097766052656008239907237112e-02 4.00627699475470944179278376395814e-01 2.01108808739028366296963667991804e-03
            5.00000000000000000000000000000000e-01 1.23311195578080790014041667745914e-02 4.87668880442191920998595833225409e-01 2.48800419844458046980673060488698e-03
            4.10757909252076280903764882168616e-01 1.45320293390108889325373908718575e-02 5.74710061408912786795610827539349e-01 2.88495418493539459298125038344551e-03
            3.24384118273061905490806111629354e-01 1.66622004254575913073654902518683e-02 6.58953681301480465037911926629022e-01 3.14750131495105008039292471266890e-03
            2.43654731456761419750023378583137e-01 1.86531678667782654934548247638304e-02 7.37692100676460293939840084931348e-01 3.23084431280764444727182649330643e-03
            1.71164420391654581443674487672979e-01 2.04409412518313272910663869197379e-02 8.08394638356514039223554846103070e-01 3.10589531739592812631389762145773e-03
            1.09242998051599260467270369190373e-01 2.19680621759608292464793777298837e-02 8.68788939772439983144636244105641e-01 2.76358496010472697718141077416476e-03
            5.98804231365069883352703072887380e-02 2.31854538018793493447944342733535e-02 9.16934123061613703953298681881279e-01 2.21683240587658859227504493105698e-03
            2.46622391156161580028083335491829e-02 2.40540130774203296526447815040228e-02 9.51283747806963564386251164251007e-01 1.49995214231954257858014933901813e-03
            4.71226234279131794835393520770594e-03 2.45460241749427265922633978334488e-02 9.70741713482266010970533898216672e-01 6.66471299188429642211539682961075e-04
            9.95287737657208571029343602276640e-01 2.22054163872896432100050717606265e-05 4.69005692640413890392991902444919e-03 1.37395558340913239250315192041541e-06
            9.75337760884383841997191666450817e-01 1.16214940673433335003791877504398e-04 2.45460241749427231228164458798346e-02 1.65144951780180537694725506625559e-05
            9.40119576863492900642427230195608e-01 2.82172263016572356323502512864820e-04 5.95982508734905283565019828984077e-02 6.14815856611750907517385811296151e-05
            8.90757001948400684021578399551800e-01 5.14781665932176769190364851169761e-04 1.08728216385667139642912104591232e-01 1.47576640711612845288658313336327e-04
            8.28835579608345418556325512327021e-01 8.06571652637296269625077904663613e-04 1.70357848739017297967635045097268e-01 2.79281620774759651790186687492223e-04
            7.56345268543238580249976621416863e-01 1.14816501568662803108022973930247e-03 2.42506566441074794537868797306146e-01 4.53190940909960909583481525331194e-04
            6.75615881726938205531496350886300e-01 1.52858306513771404107349116685555e-03 3.22855535207924093654696662269998e-01 6.58015445203858012970798974095032e-04
            5.89242090747923774607386349089211e-01 1.93559902777225203290845101378181e-03 4.08822310224303986370131269723061e-01 8.75672130250865080394739869973364e-04
            5.00000000000000000000000000000000e-01 2.35613117139565897417696760385297e-03 4.97643868828604341025823032396147e-01 1.08333192871340141562319647761115e-03
            4.10757909252076280903764882168616e-01 2.77666331501906504808374620552058e-03 5.86465427432904640170363563811406e-01 1.25617271199531584796593719488556e-03
            3.24384118273061905490806111629354e-01 3.18367927765360325675914054954774e-03 6.72432202449284477374646940006642e-01 1.37049152581236732828529056860134e-03
            2.43654731456761419750023378583137e-01 3.56409732710469013411413996550436e-03 7.52781171216133859758201651857235e-01 1.40678090613945280459851083776357e-03
            1.71164420391654581443674487672979e-01 3.90569069015402189556929180014322e-03 8.24929888918191411839586635323940e-01 1.35237535639206891274755051313150e-03
            1.09242998051599260467270369190373e-01 4.19748067685914161284443935073796e-03 8.86559521271541695064399846160086e-01 1.20332587335072798903623514377159e-03
            5.98804231365069883352703072887380e-02 4.43009007977474607992141031331812e-03 9.35689486783718216145189217058942e-01 9.65257746507837714397548545264272e-04
            2.46622391156161580028083335491829e-02 4.59604740211788480308552351516482e-03 9.70741713482266010970533898216672e-01 6.53112260957072429147507452995569e-04
            4.71226234279131794835393520770594e-03 4.69005692640402874898919449719870e-03 9.90597680730804674986700320005184e-01 2.90196310132155306082263335198945e-04
            ];
        
    case  33
        
        % ALG. DEG.:   33
        % PTS CARD.:  324
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.2e-15
        
        xyw_bar=[
            9.95782584210465393503852737921989e-01 4.19962919359279096553905219479930e-03 1.77865959418155306082098832121119e-05 9.85298067232849771816382417055635e-07
            9.77911974785698800971545097127091e-01 2.19948708280027112327292826421399e-02 9.31543862984877957256202307689819e-05 1.18682312590708857844446627582968e-05
            9.46301233248777906581494789861608e-01 5.34722967244469563063979933303926e-02 2.26470026775137112107216807999066e-04 4.43557080935164842541706953227276e-05
            9.01852479486261682239955916884355e-01 9.77335916110200625395165729969449e-02 4.13928902718255220527510118699865e-04 1.07077207217524340214336031618103e-04
            8.45843521530176500888842383574229e-01 1.53506336503465762310938202972466e-01 6.50141966357736800219413453305606e-04 2.04192203324089023488280281171114e-04
            7.79885415536973658490182970126625e-01 2.19186269739005068490911298795254e-01 9.28314724021273018905731078120880e-04 3.34589482407515238261697110289106e-04
            7.05875580731421203850572965166066e-01 2.92883974298667748570323965395801e-01 1.24044496991104757910306943813339e-03 4.91726914631658009943726561630228e-04
            6.25943112845752769324292330566095e-01 3.72479333732178730098638652634691e-01 1.57755342206850057706901679921430e-03 6.64138507134097509469750075794536e-04
            5.42387506520867757053849800286116e-01 4.55682551323644990937111742823618e-01 1.92994215548725200903845689026639e-03 8.36557747131437914357710017299041e-04
            4.57612493479132464990755124745192e-01 5.40100032886820069499833607551409e-01 2.28747363404746550941126770339906e-03 9.91534271885054262554715087674140e-04
            3.74056887154247230675707669433905e-01 6.23303250478286607894062854029471e-01 2.63986236746616143022947653662413e-03 1.11136284023244207019853746487570e-03
            2.94124419268578907171729497349588e-01 7.02898609911797533911226310010534e-01 2.97697081962355891704419263987802e-03 1.18010610030288231285522382307818e-03
            2.20114584463026341509817029873375e-01 7.76596314471460269501790207868908e-01 3.28910106551338898839276225771755e-03 1.18548009055491972325857563674845e-03
            1.54156478469823443600006385167944e-01 8.42276247706999603437338919320609e-01 3.56727382317689745150346425361931e-03 1.12038529968406163836447397841312e-03
            9.81475205137383732711953143734718e-02 8.98048992599445372597699588368414e-01 3.80348688681630964225632851594128e-03 9.83905088178671158316568323698448e-04
            5.36987667512220934185052101383917e-02 9.42310287486018416380773032869911e-01 3.99094576275949020072175699169748e-03 7.81654101387017712608551445185867e-04
            2.20880252143010324950012090994278e-02 9.73787713382462838396236293192487e-01 4.12426140323618461991372896591201e-03 5.25446949429276026141688582526967e-04
            4.21741578953443996269356830453034e-03 9.91582955016872658049464916985016e-01 4.19962919359284647669028345262632e-03 2.32640722321326146775424814627797e-04
            9.95782584210465393503852737921989e-01 4.12426140323617421157287310506945e-03 9.31543862984322845743889729419607e-05 2.26608152699165567349687613885045e-06
            9.77911974785698800971545097127091e-01 2.16001443564335406100074266078082e-02 4.87880857867658418447476265100704e-04 2.72956788495248945259491479831127e-05
            9.46301233248777906581494789861608e-01 5.25126670372442205803054093848914e-02 1.18609971397787283819980075350031e-03 1.02013445545101410312158196624921e-04
            9.01852479486261682239955916884355e-01 9.59796356059097743873564922978403e-02 2.16788490782854337268759081780445e-03 2.46266271402465364986911211531151e-04
            8.45843521530176500888842383574229e-01 1.50751466286434099295377109228866e-01 3.40501218338939981578050719690509e-03 4.69620509058698209700621895024142e-04
            7.79885415536973658490182970126625e-01 2.15252687971371536868048224278027e-01 4.86189649165480464176880559534766e-03 7.69520483622533396292353380374607e-04
            7.05875580731421203850572965166066e-01 2.87627791679632649479714245899231e-01 6.49662758894614666971278893470298e-03 1.13091998718806868197417347943201e-03
            6.25943112845752769324292330566095e-01 3.65794709199201206750728943006834e-01 8.26217795504602392497872642707080e-03 1.52744844674979800627778736554774e-03
            5.42387506520867757053849800286116e-01 4.47504737184785894577032649976900e-01 1.01077562943463483691175497369841e-02 1.92399449474237819318611641961070e-03
            4.57612493479132464990755124745192e-01 5.30407237600912684349907522118883e-01 1.19802689199548506593373531359248e-02 2.28042413927403682730177258974891e-03
            3.74056887154247230675707669433905e-01 6.12117265586497594220816154120257e-01 1.38258472592551751034761764458381e-02 2.55601719498811143227445796810571e-03
            2.94124419268578907171729497349588e-01 6.90284183106065984958377157454379e-01 1.55913976253551078698933451960329e-02 2.71411943524551895964846970343842e-03
            2.20114584463026341509817029873375e-01 7.62659286814327264103496872849064e-01 1.72261287226463943866860972775612e-02 2.72647904544000199006381635058460e-03
            1.54156478469823443600006385167944e-01 8.27160508499264590653865525382571e-01 1.86830130309119102349768581916578e-02 2.57676789913672144721745738138452e-03
            9.81475205137383732711953143734718e-02 8.81932339179789082095339836087078e-01 1.99201403064726001446160807972774e-02 2.26287782223759522878814287594196e-03
            5.36987667512220934185052101383917e-02 9.25399307748454580391239687742200e-01 2.09019255003233261902551021194085e-02 1.79772190625010594607069425165946e-03
            2.20880252143010324950012090994278e-02 9.56311830429265374853287084988551e-01 2.16001443564336481628629371698480e-02 1.20847250706562965062718095765604e-03
            4.21741578953443996269356830453034e-03 9.73787713382462727373933830676833e-01 2.19948708280027771522213697608095e-02 5.35049099161347000222499481481009e-04
            9.95782584210465393503852737921989e-01 3.99094576275946678195483130480170e-03 2.26470026775139714192430773209708e-04 3.48362681281366688641797438319436e-06
            9.77911974785698800971545097127091e-01 2.09019255003232741485508228151957e-02 1.18609971397792487990408005771314e-03 4.19614023509516759452390999296512e-05
            9.46301233248777906581494789861608e-01 5.08152092006199374019104197941488e-02 2.88355755060215601659479034424294e-03 1.56824355141451526624712942137307e-04
            9.01852479486261682239955916884355e-01 9.28771197024603478720194971174351e-02 5.27040081127796988802458599820966e-03 3.78582930901071208762420639004631e-04
            8.45843521530176500888842383574229e-01 1.45878465689282615924682318109262e-01 8.27801278054088318647529831650900e-03 7.21943397762894625672991377740573e-04
            7.79885415536973658490182970126625e-01 2.08294702733404057948263243815745e-01 1.18198817296222835615537860576296e-02 1.18297693963182316660709414435360e-03
            7.05875580731421203850572965166066e-01 2.78330300682436693193722021533176e-01 1.57941185861421029557050133007579e-02 1.73855315600467867989675774254010e-03
            6.25943112845752769324292330566095e-01 3.53970493619263082418768817660748e-01 2.00863935349841482569388517731568e-02 2.34813280144963599377350149666199e-03
            5.42387506520867757053849800286116e-01 4.33039266929351129853387192270020e-01 2.45732265497811130927630074438639e-02 2.95773948543163458721472203194480e-03
            4.57612493479132464990755124745192e-01 5.13261966319426554683502672560280e-01 2.91255402014409803257422026945278e-02 3.50567558207353713664589101028923e-03
            3.74056887154247230675707669433905e-01 5.92330739629514768651574740943033e-01 3.36123732162380006727175896230619e-02 3.92934231554068445768646000715307e-03
            2.94124419268578907171729497349588e-01 6.67970932566341102365470305812778e-01 3.79046481650799904628001968376338e-02 4.17239147187787543652470745314531e-03
            2.20114584463026341509817029873375e-01 7.38006530515373793122080314788036e-01 4.18788850215998653681026553385891e-02 4.19139178980844264582605873670218e-03
            1.54156478469823443600006385167944e-01 8.00422767559495262901236856123433e-01 4.54207539706812379876055274507962e-02 3.96124218696888939078570857077466e-03
            9.81475205137383732711953143734718e-02 8.53424113546317641976202139630914e-01 4.84283659399440402637537772534415e-02 3.47870178622100090573554531658829e-03
            5.36987667512220934185052101383917e-02 8.95486024048157913668433138809633e-01 5.08152092006199929130616510519758e-02 2.76362176735507484096054930944319e-03
            2.20880252143010324950012090994278e-02 9.25399307748454802435844612773508e-01 5.25126670372442205803054093848914e-02 1.85777394944426567542639094909873e-03
            4.21741578953443996269356830453034e-03 9.42310287486018527403075495385565e-01 5.34722967244469771230797050520778e-02 8.22526182667712301051221590597606e-04
            9.95782584210465393503852737921989e-01 3.80348688681629489710678271308097e-03 4.13928902718311599040479364930434e-04 4.60112595870666033420282384214239e-06
            9.77911974785698800971545097127091e-01 1.99201403064725515723587534466787e-02 2.16788490782864745609614942623011e-03 5.54220380066369058889678977930515e-05
            9.46301233248777906581494789861608e-01 4.84283659399440749582232967895834e-02 5.27040081127801846028191334880830e-03 2.07131432317793535719668840400232e-04
            9.01852479486261682239955916884355e-01 8.85145847307436811624015149391198e-02 9.63293578299463659764256817652495e-03 5.00027082259496850544233659263682e-04
            8.45843521530176500888842383574229e-01 1.39026402336880794408102701709140e-01 1.51300761329427047030549147166312e-02 9.53532822731037151345323543694121e-04
            7.79885415536973658490182970126625e-01 1.98510883769068419146819337584020e-01 2.16037006939579223629976922893547e-02 1.56245952794670977040181192307955e-03
            7.05875580731421203850572965166066e-01 2.65256836794824535363801487619639e-01 2.88675824737542607856255472142948e-02 2.29625688585845255662576391841867e-03
            6.25943112845752769324292330566095e-01 3.37344131148970660660779685713351e-01 3.67127560052765700149279837205540e-02 3.10138122358584586088503520784343e-03
            5.42387506520867757053849800286116e-01 4.12698961888046111923245007346850e-01 4.49135315910861310229051923670340e-02 3.90654127343780140091400454593895e-03
            4.57612493479132464990755124745192e-01 4.89153517598215292760954753248370e-01 5.32339889226522422482901220064377e-02 4.63024766721629923299508035938743e-03
            3.74056887154247230675707669433905e-01 5.64508348337291021579176231171004e-01 6.14347645084617477451160993950907e-02 5.18982081036296546971975374162867e-03
            2.94124419268578907171729497349588e-01 6.36595642691436980342700735491235e-01 6.92799380399841124855697671591770e-02 5.51083676372266829945889554664973e-03
            2.20114584463026341509817029873375e-01 7.03341595717193235337560963671422e-01 7.65438198197804231526220064552035e-02 5.53593212001411878742107575135378e-03
            1.54156478469823443600006385167944e-01 7.62826077149380887831853215175215e-01 8.30174443807956130569891683990136e-02 5.23195371793154853645724244870507e-03
            9.81475205137383732711953143734718e-02 8.13337894755518098222069056646433e-01 8.85145847307435840178868602379225e-02 4.59462105191828489719751971165351e-03
            5.36987667512220934185052101383917e-02 8.53424113546317641976202139630914e-01 9.28771197024602646052926502306946e-02 3.65015328480432048319226367993906e-03
            2.20880252143010324950012090994278e-02 8.81932339179789304139944761118386e-01 9.59796356059097188762052610400133e-02 2.45372205563346460974183393943804e-03
            4.21741578953443996269356830453034e-03 8.98048992599445483620002050884068e-01 9.77335916110200209061531495535746e-02 1.08638116943748984193074402782031e-03
            9.95782584210465393503852737921989e-01 3.56727382317692130395125893471686e-03 6.50141966357685192196003143294547e-04 5.58629407564891192932484548228089e-06
            9.77911974785698800971545097127091e-01 1.86830130309118096210152515368463e-02 3.40501218338938940743965133606252e-03 6.72887039727753337578150527598098e-05
            9.46301233248777906581494789861608e-01 4.54207539706812518653933352652530e-02 8.27801278054084155311187487313873e-03 2.51481290367198343903981649916091e-04
            9.01852479486261682239955916884355e-01 8.30174443807957102015038231002109e-02 1.51300761329426075585402600154339e-02 6.07090167137135420204197000515478e-04
            8.45843521530176500888842383574229e-01 1.30392258615606299043676585824869e-01 2.37642198542172000674810306009022e-02 1.15769809528458856966171008906485e-03
            7.79885415536973658490182970126625e-01 1.86182495262357639020933675055858e-01 3.39320892006687024888833548175171e-02 1.89700488157541606072664652771209e-03
            7.05875580731421203850572965166066e-01 2.48783234562152733015949479522533e-01 4.53411847064260631334775553114014e-02 2.78791894696245583923355226829699e-03
            6.25943112845752769324292330566095e-01 3.16393594683164303127398397919023e-01 5.76632924710829275483092715148814e-02 3.76543213794485387077570770486545e-03
            5.42387506520867757053849800286116e-01 3.87068562980594110811694008589257e-01 7.05439304985381321344561911246274e-02 4.74298868753809162862600246057809e-03
            4.57612493479132464990755124745192e-01 4.58774958549582168032543449953664e-01 8.36125479712853669767014253011439e-02 5.62165116632197363333300899057576e-03
            3.74056887154247230675707669433905e-01 5.29449926847012197761443985655205e-01 9.64931859987405715628483449108899e-02 6.30103707370781307428586615060340e-03
            2.94124419268578907171729497349588e-01 5.97060286968023601339439210278215e-01 1.08815293763397491488831292372197e-01 6.69078722063623423238487575304134e-03
            2.20114584463026341509817029873375e-01 6.59661026267818861867908708518371e-01 1.20224389269154796622274261608254e-01 6.72125585840781074320471333294336e-03
            1.54156478469823443600006385167944e-01 7.15451262914570174089590182120446e-01 1.30392258615606326799252201453783e-01 6.35219125076198765089685593920876e-03
            9.81475205137383732711953143734718e-02 7.62826077149380887831853215175215e-01 1.39026402336880794408102701709140e-01 5.57839637352541654891080469269582e-03
            5.36987667512220934185052101383917e-02 8.00422767559495262901236856123433e-01 1.45878465689282643680257933738176e-01 4.43170429436461066341479408947635e-03
            2.20880252143010324950012090994278e-02 8.27160508499264812698470450413879e-01 1.50751466286434210317679571744520e-01 2.97909970422267694409312888126351e-03
            4.21741578953443996269356830453034e-03 8.42276247706999714459641381836263e-01 1.53506336503465790066513818601379e-01 1.31899120893249569702598833487173e-03
            9.95782584210465393503852737921989e-01 3.28910106551339028943536924032287e-03 9.28314724021216206711892837688538e-04 6.41076541820288155643656549664833e-06
            9.77911974785698800971545097127091e-01 1.72261287226463978561330492311754e-02 4.86189649165480117232185364173347e-03 7.72197257471179948759218425458073e-05
            9.46301233248777906581494789861608e-01 4.18788850215998653681026553385891e-02 1.18198817296222280504025547998026e-02 2.88596972837273804766283769041024e-04
            9.01852479486261682239955916884355e-01 7.65438198197804925415610455274873e-02 2.16037006939578252184830375881575e-02 6.96689539882782587522369599497551e-04
            8.45843521530176500888842383574229e-01 1.20224389269154782744486453793797e-01 3.39320892006687163666711626319739e-02 1.32856072621054515509042559529007e-03
            7.79885415536973658490182970126625e-01 1.71664154169695548635132809067727e-01 4.84504302933307928746842208056478e-02 2.17698050411946168736232465334979e-03
            7.05875580731421203850572965166066e-01 2.29383344940846584103510963359440e-01 6.47410743277322120459160714744939e-02 3.19938301348079122410350549898794e-03
            6.25943112845752769324292330566095e-01 2.91721510872756939480865412406274e-01 8.23353762814902911948422570276307e-02 4.32116566146270326270739658980347e-03
            5.42387506520867757053849800286116e-01 3.56885309631883640424376835653675e-01 1.00727183847248602521773364060209e-01 5.44299806727676499518153718781832e-03
            4.57612493479132464990755124745192e-01 4.23000105905089796021201209441642e-01 1.19387400615777738988043665813166e-01 6.45134079986121270616772349626444e-03
            3.74056887154247230675707669433905e-01 4.88163904664216719009317557720351e-01 1.37779208181536050314974772845744e-01 7.23099608146713073580924557859362e-03
            2.94124419268578907171729497349588e-01 5.50502070596126880097642697364790e-01 1.55373510135294212730627805285621e-01 7.67826876883956777747330946226612e-03
            2.20114584463026341509817029873375e-01 6.08221261367278054343898929801071e-01 1.71664154169695604146284040325554e-01 7.71323421940852804040700263499275e-03
            1.54156478469823443600006385167944e-01 6.59661026267818861867908708518371e-01 1.86182495262357639020933675055858e-01 7.28970001377263459446664839447294e-03
            9.81475205137383732711953143734718e-02 7.03341595717193235337560963671422e-01 1.98510883769068446902394953212934e-01 6.40170210807796801616431636716698e-03
            5.36987667512220934185052101383917e-02 7.38006530515373793122080314788036e-01 2.08294702733404113459414475073572e-01 5.08577175660299141135567424498731e-03
            2.20880252143010324950012090994278e-02 7.62659286814327375125799335364718e-01 2.15252687971371647890350686793681e-01 3.41877980331543570094443573736953e-03
            4.21741578953443996269356830453034e-03 7.76596314471460380524092670384562e-01 2.19186269739005124002062530053081e-01 1.51365880754421680033650066121709e-03
            9.95782584210465393503852737921989e-01 2.97697081962360575457804401366957e-03 1.24044496991100074156921806434184e-03 7.05081495465016400267434931725674e-06
            9.77911974785698800971545097127091e-01 1.55913976253550939921055373815761e-02 6.49662758894610503634936549133272e-03 8.49293277126948574365719601964031e-05
            9.46301233248777906581494789861608e-01 3.79046481650800251572697163737757e-02 1.57941185861420682612354937646160e-02 3.17410436852054054709948438528500e-04
            9.01852479486261682239955916884355e-01 6.92799380399841541189331906025473e-02 2.88675824737541636411108925130975e-02 7.66246884124938457648334999561257e-04
            8.45843521530176500888842383574229e-01 1.08815293763397449855467868928827e-01 4.53411847064260492556897474969446e-02 1.46120396324720780260875141465249e-03
            7.79885415536973658490182970126625e-01 1.55373510135294129463900958398881e-01 6.47410743277322120459160714744939e-02 2.39432980199893920975062755474028e-03
            7.05875580731421203850572965166066e-01 2.07615245258500030756110277252446e-01 8.65091740100787653933167575814878e-02 3.51880877329431053512909244318507e-03
            6.25943112845752769324292330566095e-01 2.64037622446591957459105515226838e-01 1.10019264707655273216602154207067e-01 4.75258997636244292367413422084610e-03
            5.42387506520867757053849800286116e-01 3.23017484584536107572461105519324e-01 1.34595008894596135373689094194560e-01 5.98642590507472878436079355424226e-03
            4.57612493479132464990755124745192e-01 3.82858096146884874233506934615434e-01 1.59529410373982660775737940639374e-01 7.09544137429342098438711516905641e-03
            3.74056887154247230675707669433905e-01 4.41837958284829246391467449939228e-01 1.84105154560923522932824880626868e-01 7.95293728319223895695255777127386e-03
            2.94124419268578907171729497349588e-01 4.98260335472921034316584609769052e-01 2.07615245258500058511685892881360e-01 8.44486558616486596151595733772410e-03
            2.20114584463026341509817029873375e-01 5.50502070596126991119945159880444e-01 2.29383344940846667370237810246181e-01 8.48332198032143366961399522097054e-03
            1.54156478469823443600006385167944e-01 5.97060286968023712361741672793869e-01 2.48783234562152788527100710780360e-01 8.01750220435144302089103263142533e-03
            9.81475205137383732711953143734718e-02 6.36595642691437091365003198006889e-01 2.65256836794824590874952718877466e-01 7.04084676545610639586403323164632e-03
            5.36987667512220934185052101383917e-02 6.67970932566341213387772768328432e-01 2.78330300682436693193722021533176e-01 5.59353419103050721744585516148618e-03
            2.20880252143010324950012090994278e-02 6.90284183106066207002982082485687e-01 2.87627791679632816013167939672712e-01 3.76011009472091784280878279389526e-03
            4.21741578953443996269356830453034e-03 7.02898609911797644933528772526188e-01 2.92883974298667859592626427911455e-01 1.66478219997987539162387360391904e-03
            9.95782584210465393503852737921989e-01 2.63986236746611979686605309325387e-03 1.57755342206848669928120898475754e-03 7.48802741672358854359281554735617e-06
            9.77911974785698800971545097127091e-01 1.38258472592551334701127530024678e-02 8.26217795504606555834214987044106e-03 9.01956920564390648466235322899820e-05
            9.46301233248777906581494789861608e-01 3.36123732162379659782480700869201e-02 2.00863935349841274402571400514717e-02 3.37092672093862813238046571839845e-04
            9.01852479486261682239955916884355e-01 6.14347645084617755006917150240042e-02 3.67127560052765422593523680916405e-02 8.13760921710537823807463109204718e-04
            8.45843521530176500888842383574229e-01 9.64931859987405576850605370964331e-02 5.76632924710829414260970793293382e-02 1.55181144429327008867069981334907e-03
            7.79885415536973658490182970126625e-01 1.37779208181536050314974772845744e-01 8.23353762814902911948422570276307e-02 2.54279928169466205509219314251368e-03
            7.05875580731421203850572965166066e-01 1.84105154560923495177249264997954e-01 1.10019264707655300972177769835980e-01 3.73700582671760610578570727113856e-03
            6.25943112845752769324292330566095e-01 2.34138332326721987364237520523602e-01 1.39918554827525243311470148910303e-01 5.04729230200218811214218916916252e-03
            5.42387506520867757053849800286116e-01 2.86439388645434722668881022400456e-01 1.71173104833697520277269177313428e-01 6.35763689640157607374515436049478e-03
            4.57612493479132464990755124745192e-01 3.39503724200317880121957614392159e-01 2.02883782320549654887287260862649e-01 7.53542106638652956634993884676987e-03
            3.74056887154247230675707669433905e-01 3.91804780519030781960054810042493e-01 2.34138332326721987364237520523602e-01 8.44608925394517602580624071606508e-03
            2.94124419268578907171729497349588e-01 4.41837958284829190880316218681401e-01 2.64037622446591901947954283969011e-01 8.96852143283705625920187998190158e-03
            2.20114584463026341509817029873375e-01 4.88163904664216719009317557720351e-01 2.91721510872756939480865412406274e-01 9.00936246123516977502010405487454e-03
            1.54156478469823443600006385167944e-01 5.29449926847012197761443985655205e-01 3.16393594683164303127398397919023e-01 8.51465776735933939367573231038477e-03
            9.81475205137383732711953143734718e-02 5.64508348337291021579176231171004e-01 3.37344131148970660660779685713351e-01 7.47744111223818102435201282673916e-03
            5.36987667512220934185052101383917e-02 5.92330739629514768651574740943033e-01 3.53970493619263137929920048918575e-01 5.94038244489645574086811308234246e-03
            2.20880252143010324950012090994278e-02 6.12117265586497705243118616635911e-01 3.65794709199201317773031405522488e-01 3.99326994968149037723570415892027e-03
            4.21741578953443996269356830453034e-03 6.23303250478286718916365316545125e-01 3.72479333732178785609789883892518e-01 1.76801331994979769521303403223556e-03
            9.95782584210465393503852737921989e-01 2.28747363404741216666438141658091e-03 1.92994215548719432948288066143050e-03 7.70982414572235148547598576618256e-06
            9.77911974785698800971545097127091e-01 1.19802689199548523940608291127319e-02 1.01077562943463466343940737601770e-02 9.28673047996321385190096608930332e-05
            9.46301233248777906581494789861608e-01 2.91255402014410288979995300451264e-02 2.45732265497810645205056800932653e-02 3.47077418126294544656329499332514e-04
            9.01852479486261682239955916884355e-01 5.32339889226522769427596415425796e-02 4.49135315910860408172844415730651e-02 8.37864667674303119880574453048894e-04
            8.45843521530176500888842383574229e-01 8.36125479712853669767014253011439e-02 7.05439304985381321344561911246274e-02 1.59777638047905296524164864280237e-03
            7.79885415536973658490182970126625e-01 1.19387400615777766743619281442079e-01 1.00727183847248574766197748431296e-01 2.61811745720249861718587069958630e-03
            7.05875580731421203850572965166066e-01 1.59529410373982688531313556268287e-01 1.34595008894596107618113478565647e-01 3.84769661649277854367778850530613e-03
            6.25943112845752769324292330566095e-01 2.02883782320549765909589723378303e-01 1.71173104833697464766117946055601e-01 5.19679401461404545659039655447486e-03
            5.42387506520867757053849800286116e-01 2.48203299290943368848871841692016e-01 2.09409194188188874097278358021867e-01 6.54595125334883100320926274662270e-03
            4.57612493479132464990755124745192e-01 2.94184207229924221671524264820619e-01 2.48203299290943313337720610434189e-01 7.75862160387657488064272470751348e-03
            3.74056887154247230675707669433905e-01 3.39503724200317991144260076907813e-01 2.86439388645434778180032253658283e-01 8.69626394817410094950638210775651e-03
            2.94124419268578907171729497349588e-01 3.82858096146884985255809397131088e-01 3.23017484584536107572461105519324e-01 9.23417066287538987334748696866882e-03
            2.20114584463026341509817029873375e-01 4.23000105905089962554654903215123e-01 3.56885309631883695935528066911502e-01 9.27622140993551204390943354383126e-03
            1.54156478469823443600006385167944e-01 4.58774958549582334565997143727145e-01 3.87068562980594166322845239847084e-01 8.76686347338098345693957469393354e-03
            9.81475205137383732711953143734718e-02 4.89153517598215514805559678279678e-01 4.12698961888046167434396238604677e-01 7.69892427297972671562664714883795e-03
            5.36987667512220934185052101383917e-02 5.13261966319426776728107597591588e-01 4.33039266929351129853387192270020e-01 6.11633764937889387591773981966981e-03
            2.20880252143010324950012090994278e-02 5.30407237600913017416814909665845e-01 4.47504737184786005599335112492554e-01 4.11155132921681588237250437600778e-03
            4.21741578953443996269356830453034e-03 5.40100032886820402566740995098371e-01 4.55682551323645101959414205339272e-01 1.82038219487075641350726229461543e-03
            9.95782584210465393503852737921989e-01 1.92994215548719541368505314693493e-03 2.28747363404741086562177443397559e-03 7.70982414572235656767366929198460e-06
            9.77911974785698800971545097127091e-01 1.01077562943462963274132704327712e-02 1.19802689199549027010416324401376e-02 9.28673047996322062816454412370604e-05
            9.46301233248777906581494789861608e-01 2.45732265497810749288465359541078e-02 2.91255402014410184896586741842839e-02 3.47077418126294869916981244983845e-04
            9.01852479486261682239955916884355e-01 4.49135315910861240840112884598057e-02 5.32339889226521936760327946558391e-02 8.37864667674303770401877944351554e-04
            8.45843521530176500888842383574229e-01 7.05439304985381043788805754957139e-02 8.36125479712853947322770409300574e-02 1.59777638047905426628425562540770e-03
            7.79885415536973658490182970126625e-01 1.00727183847248574766197748431296e-01 1.19387400615777766743619281442079e-01 2.61811745720250078559021567059517e-03
            7.05875580731421203850572965166066e-01 1.34595008894596107618113478565647e-01 1.59529410373982688531313556268287e-01 3.84769661649278201312474045892031e-03
            6.25943112845752769324292330566095e-01 1.71173104833697548032844792942342e-01 2.02883782320549682642862876491563e-01 5.19679401461404979339908649649260e-03
            5.42387506520867757053849800286116e-01 2.09409194188188929608429589279694e-01 2.48203299290943313337720610434189e-01 6.54595125334883620737969067704398e-03
            4.57612493479132464990755124745192e-01 2.48203299290943424360023072949843e-01 2.94184207229924110649221802304965e-01 7.75862160387658095217489062633831e-03
            3.74056887154247230675707669433905e-01 2.86439388645434889202334716173937e-01 3.39503724200317880121957614392159e-01 8.69626394817410962312376199179198e-03
            2.94124419268578907171729497349588e-01 3.23017484584536274105914799292805e-01 3.82858096146884818722355703357607e-01 9.23417066287539681224139087589720e-03
            2.20114584463026341509817029873375e-01 3.56885309631883862468981760684983e-01 4.23000105905089796021201209441642e-01 9.27622140993551898280333745105963e-03
            1.54156478469823443600006385167944e-01 3.87068562980594332856298933620565e-01 4.58774958549582168032543449953664e-01 8.76686347338099039583347860116191e-03
            9.81475205137383732711953143734718e-02 4.12698961888046389479001163635985e-01 4.89153517598215292760954753248370e-01 7.69892427297973278715881306766278e-03
            5.36987667512220934185052101383917e-02 4.33039266929351407409143348559155e-01 5.13261966319426443661200210044626e-01 6.11633764937889908008816775009109e-03
            2.20880252143010324950012090994278e-02 4.47504737184786227643940037523862e-01 5.30407237600912795372209984634537e-01 4.11155132921681935181945632962197e-03
            4.21741578953443996269356830453034e-03 4.55682551323645324004019130370580e-01 5.40100032886820180522136070067063e-01 1.82038219487075793139030377432164e-03
            9.95782584210465393503852737921989e-01 1.57755342206848669928120898475754e-03 2.63986236746611979686605309325387e-03 7.48802741672362666007544199087143e-06
            9.77911974785698800971545097127091e-01 8.26217795504601004719091861261404e-03 1.38258472592551889812639842602948e-02 9.01956920564395256325468386293664e-05
            9.46301233248777906581494789861608e-01 2.00863935349841274402571400514717e-02 3.36123732162379659782480700869201e-02 3.37092672093864493751413924371718e-04
            9.01852479486261682239955916884355e-01 3.67127560052765977705035993494675e-02 6.14347645084617199895404837661772e-02 8.13760921710541943775718554121568e-04
            8.45843521530176500888842383574229e-01 5.76632924710828928538397519787395e-02 9.64931859987405993184239605398034e-02 1.55181144429327811176677620608189e-03
            7.79885415536973658490182970126625e-01 8.23353762814902495614788335842604e-02 1.37779208181536105826126004103571e-01 2.54279928169467506551826296856689e-03
            7.05875580731421203850572965166066e-01 1.10019264707655259338814346392610e-01 1.84105154560923522932824880626868e-01 3.73700582671762518774394301601660e-03
            6.25943112845752769324292330566095e-01 1.39918554827525243311470148910303e-01 2.34138332326721987364237520523602e-01 5.04729230200221413299432882126894e-03
            5.42387506520867757053849800286116e-01 1.71173104833697437010542330426688e-01 2.86439388645434833691183484916110e-01 6.35763689640160903349119791982957e-03
            4.57612493479132464990755124745192e-01 2.02883782320549682642862876491563e-01 3.39503724200317824610806383134332e-01 7.53542106638656773026641033652595e-03
            3.74056887154247230675707669433905e-01 2.34138332326721987364237520523602e-01 3.91804780519030781960054810042493e-01 8.44608925394521939389314013624244e-03
            2.94124419268578907171729497349588e-01 2.64037622446591901947954283969011e-01 4.41837958284829190880316218681401e-01 8.96852143283710136201225537888604e-03
            2.20114584463026341509817029873375e-01 2.91721510872756939480865412406274e-01 4.88163904664216719009317557720351e-01 9.00936246123521487783047945185899e-03
            1.54156478469823443600006385167944e-01 3.16393594683164303127398397919023e-01 5.29449926847012197761443985655205e-01 8.51465776735938276176263173056213e-03
            9.81475205137383732711953143734718e-02 3.37344131148970660660779685713351e-01 5.64508348337291021579176231171004e-01 7.47744111223821832090674632809169e-03
            5.36987667512220934185052101383917e-02 3.53970493619263082418768817660748e-01 5.92330739629514768651574740943033e-01 5.94038244489648609852894267646661e-03
            2.20880252143010324950012090994278e-02 3.65794709199201262261880174264661e-01 6.12117265586497705243118616635911e-01 3.99326994968151119391741588060540e-03
            4.21741578953443996269356830453034e-03 3.72479333732178785609789883892518e-01 6.23303250478286718916365316545125e-01 1.76801331994980658567084841337191e-03
            9.95782584210465393503852737921989e-01 1.24044496991100139209052155564450e-03 2.97697081962360488721630602526602e-03 7.05081495465018179036624165756386e-06
            9.77911974785698800971545097127091e-01 6.49662758894605212728334819871634e-03 1.55913976253551460338098166857890e-02 8.49293277126950742770064572972899e-05
            9.46301233248777906581494789861608e-01 1.57941185861420717306824457182302e-02 3.79046481650800182183758124665474e-02 3.17410436852054867861577802656825e-04
            9.01852479486261682239955916884355e-01 2.88675824737542330300499315853813e-02 6.92799380399840847299941515302635e-02 7.66246884124940409212245473469238e-04
            8.45843521530176500888842383574229e-01 4.53411847064260076223263240535744e-02 1.08815293763397491488831292372197e-01 1.46120396324721148889613786536756e-03
            7.79885415536973658490182970126625e-01 6.47410743277321704125526480311237e-02 1.55373510135294157219476574027794e-01 2.39432980199894528128279347356511e-03
            7.05875580731421203850572965166066e-01 8.65091740100787376377411419525743e-02 2.07615245258500058511685892881360e-01 3.51880877329431964242734132142232e-03
            6.25943112845752769324292330566095e-01 1.10019264707655314849965577650437e-01 2.64037622446591901947954283969011e-01 4.75258997636245419937672807009221e-03
            5.42387506520867757053849800286116e-01 1.34595008894596107618113478565647e-01 3.23017484584536163083612336777151e-01 5.98642590507474352951033935710257e-03
            4.57612493479132464990755124745192e-01 1.59529410373982716286889171897201e-01 3.82858096146884818722355703357607e-01 7.09544137429343833162187493712736e-03
            3.74056887154247230675707669433905e-01 1.84105154560923578443976111884695e-01 4.41837958284829190880316218681401e-01 7.95293728319225977363426949295899e-03
            2.94124419268578907171729497349588e-01 2.07615245258500114022837124139187e-01 4.98260335472920978805433378511225e-01 8.44486558616488677819766905940924e-03
            2.20114584463026341509817029873375e-01 2.29383344940846722881389041504008e-01 5.50502070596126991119945159880444e-01 8.48332198032145448629570694265567e-03
            1.54156478469823443600006385167944e-01 2.48783234562152871793827557667100e-01 5.97060286968023601339439210278215e-01 8.01750220435146210284926837630337e-03
            9.81475205137383732711953143734718e-02 2.65256836794824701897255181393120e-01 6.36595642691436980342700735491235e-01 7.04084676545612374309879299971726e-03
            5.36987667512220934185052101383917e-02 2.78330300682436859727175715306657e-01 6.67970932566340991343167843297124e-01 5.59353419103052109523366297594293e-03
            2.20880252143010324950012090994278e-02 2.87627791679632927035470402188366e-01 6.90284183106066095980679619970033e-01 3.76011009472092738378790066633428e-03
            4.21741578953443996269356830453034e-03 2.92883974298667915103777659169282e-01 7.02898609911797533911226310010534e-01 1.66478219997987972843256354593677e-03
            9.95782584210465393503852737921989e-01 9.28314724021216098291675589138094e-04 3.28910106551339028943536924032287e-03 6.41076541820290527335908861705782e-06
            9.77911974785698800971545097127091e-01 4.86189649165474479380888439550290e-03 1.72261287226464533672842804890024e-02 7.72197257471182794789921199907212e-05
            9.46301233248777906581494789861608e-01 1.18198817296222263156790788229955e-02 4.18788850215998653681026553385891e-02 2.88596972837274888968456254545458e-04
            9.01852479486261682239955916884355e-01 2.16037006939578772601873168923703e-02 7.65438198197804370304098142696603e-02 6.96689539882785189607583564708193e-04
            8.45843521530176500888842383574229e-01 3.39320892006686539166260274669185e-02 1.20224389269154852133425492866081e-01 1.32856072621055014242041902861047e-03
            7.79885415536973658490182970126625e-01 4.84504302933307443024268934550491e-02 1.71664154169695604146284040325554e-01 2.17698050411946949361796654898171e-03
            7.05875580731421203850572965166066e-01 6.47410743277321426569770324022102e-02 2.29383344940846667370237810246181e-01 3.19938301348080293348696834243583e-03
            6.25943112845752769324292330566095e-01 8.23353762814902634392666413987172e-02 2.91721510872756994992016643664101e-01 4.32116566146271974258041836947086e-03
            5.42387506520867757053849800286116e-01 1.00727183847248533132834324987925e-01 3.56885309631883695935528066911502e-01 5.44299806727678581186324890950345e-03
            4.57612493479132464990755124745192e-01 1.19387400615777752865831473627622e-01 4.23000105905089796021201209441642e-01 6.45134079986123699229638717156377e-03
            3.74056887154247230675707669433905e-01 1.37779208181536078070550388474658e-01 4.88163904664216663498166326462524e-01 7.23099608146715762402312321910358e-03
            2.94124419268578907171729497349588e-01 1.55373510135294157219476574027794e-01 5.50502070596126991119945159880444e-01 7.67826876883959640041066307958317e-03
            2.20114584463026341509817029873375e-01 1.71664154169695576390708424696641e-01 6.08221261367278054343898929801071e-01 7.71323421940855666334435625230981e-03
            1.54156478469823443600006385167944e-01 1.86182495262357666776509290684771e-01 6.59661026267818861867908708518371e-01 7.28970001377266148268052603498290e-03
            9.81475205137383732711953143734718e-02 1.98510883769068474657970568841847e-01 7.03341595717193235337560963671422e-01 6.40170210807799230229298004246630e-03
            5.36987667512220934185052101383917e-02 2.08294702733404113459414475073572e-01 7.38006530515373793122080314788036e-01 5.08577175660301049331390998986535e-03
            2.20880252143010324950012090994278e-02 2.15252687971371620134775071164768e-01 7.62659286814327375125799335364718e-01 3.41877980331544871137050556342274e-03
            4.21741578953443996269356830453034e-03 2.19186269739005151757638145681995e-01 7.76596314471460380524092670384562e-01 1.51365880754422265502823208294103e-03
            9.95782584210465393503852737921989e-01 6.50141966357684975355568646193660e-04 3.56727382317692173763212792891864e-03 5.58629407564889583569884765057445e-06
            9.77911974785698800971545097127091e-01 3.40501218338933172788407510722664e-03 1.86830130309118686016134347482875e-02 6.72887039727751304699077117277284e-05
            9.46301233248777906581494789861608e-01 8.27801278054083981838839889633164e-03 4.54207539706812518653933352652530e-02 2.51481290367197639172569534338209e-04
            9.01852479486261682239955916884355e-01 1.51300761329426648044149672500680e-02 8.30174443807956546903525918423838e-02 6.07090167137133577060503775157940e-04
            8.45843521530176500888842383574229e-01 2.37642198542171376174358954358468e-02 1.30392258615606354554827817082696e-01 1.15769809528458510021475813545067e-03
            7.79885415536973658490182970126625e-01 3.39320892006686469777321235596901e-02 1.86182495262357694532084906313685e-01 1.89700488157541042287534960308903e-03
            7.05875580731421203850572965166066e-01 4.53411847064259798667507084246608e-02 2.48783234562152816282676326409273e-01 2.78791894696244759929704137846329e-03
            6.25943112845752769324292330566095e-01 5.76632924710828928538397519787395e-02 3.16393594683164358638549629176850e-01 3.76543213794484259507311385561934e-03
            5.42387506520867757053849800286116e-01 7.05439304985380627455171520523436e-02 3.87068562980594166322845239847084e-01 4.74298868753807775083819464612134e-03
            4.57612493479132464990755124745192e-01 8.36125479712853392211258096722304e-02 4.58774958549582168032543449953664e-01 5.62165116632195715345998721090837e-03
            3.74056887154247230675707669433905e-01 9.64931859987405576850605370964331e-02 5.29449926847012197761443985655205e-01 6.30103707370779485968936839412891e-03
            2.94124419268578907171729497349588e-01 1.08815293763397435977680061114370e-01 5.97060286968023712361741672793869e-01 6.69078722063621428306490201975976e-03
            2.20114584463026341509817029873375e-01 1.20224389269154782744486453793797e-01 6.59661026267818861867908708518371e-01 6.72125585840779079388473959966177e-03
            1.54156478469823443600006385167944e-01 1.30392258615606299043676585824869e-01 7.15451262914570174089590182120446e-01 6.35219125076196856893862019433072e-03
            9.81475205137383732711953143734718e-02 1.39026402336880794408102701709140e-01 7.62826077149380887831853215175215e-01 5.57839637352540006903778291302842e-03
            5.36987667512220934185052101383917e-02 1.45878465689282615924682318109262e-01 8.00422767559495262901236856123433e-01 4.43170429436459765298872426342314e-03
            2.20880252143010324950012090994278e-02 1.50751466286434127050952724857780e-01 8.27160508499264923720772912929533e-01 2.97909970422266827047574899722804e-03
            4.21741578953443996269356830453034e-03 1.53506336503465762310938202972466e-01 8.42276247706999714459641381836263e-01 1.31899120893249179389816738705576e-03
            9.95782584210465393503852737921989e-01 4.13928902718311924301131110581764e-04 3.80348688681629446342591371887920e-03 4.60112595870661882958840838142578e-06
            9.77911974785698800971545097127091e-01 2.16788490782859194494491816840309e-03 1.99201403064726070835099847045058e-02 5.54220380066363976691995452128481e-05
            9.46301233248777906581494789861608e-01 5.27040081127802279709060329082604e-03 4.84283659399440680193293928823550e-02 2.07131432317791638365866990767472e-04
            9.01852479486261682239955916884355e-01 9.63293578299469557824075138796616e-03 8.85145847307436256512502836812928e-02 5.00027082259492296895109220145059e-04
            8.45843521530176500888842383574229e-01 1.51300761329426648044149672500680e-02 1.39026402336880822163678317338054e-01 9.53532822731028477727943659658649e-04
            7.79885415536973658490182970126625e-01 2.16037006939578703212934129851419e-02 1.98510883769068474657970568841847e-01 1.56245952794669567577356961152191e-03
            7.05875580731421203850572965166066e-01 2.88675824737542191522621237709245e-02 2.65256836794824590874952718877466e-01 2.29625688585843173994405219673354e-03
            6.25943112845752769324292330566095e-01 3.67127560052765977705035993494675e-02 3.37344131148970660660779685713351e-01 3.10138122358581767162855058472815e-03
            5.42387506520867757053849800286116e-01 4.49135315910860963284356728308921e-02 4.12698961888046167434396238604677e-01 3.90654127343776583908274702139352e-03
            4.57612493479132464990755124745192e-01 5.32339889226522491871840259136661e-02 4.89153517598215292760954753248370e-01 4.63024766721625673226991892761362e-03
            3.74056887154247230675707669433905e-01 6.14347645084617755006917150240042e-02 5.64508348337291021579176231171004e-01 5.18982081036291863218590236783712e-03
            2.94124419268578907171729497349588e-01 6.92799380399841402411453827880905e-02 6.36595642691436980342700735491235e-01 5.51083676372261799247809221924399e-03
            2.20114584463026341509817029873375e-01 7.65438198197804925415610455274873e-02 7.03341595717193124315258501155768e-01 5.53593212001406934780201041235159e-03
            1.54156478469823443600006385167944e-01 8.30174443807957102015038231002109e-02 7.62826077149380776809550752659561e-01 5.23195371793150083156165308650998e-03
            9.81475205137383732711953143734718e-02 8.85145847307436811624015149391198e-02 8.13337894755517987199766594130779e-01 4.59462105191824326383409626828325e-03
            5.36987667512220934185052101383917e-02 9.28771197024603478720194971174351e-02 8.53424113546317530953899677115260e-01 3.65015328480428752344622012060427e-03
            2.20880252143010324950012090994278e-02 9.59796356059098021429321079267538e-02 8.81932339179789193117642298602732e-01 2.45372205563344205833664624094581e-03
            4.21741578953443996269356830453034e-03 9.77335916110200764173043808114016e-02 8.98048992599445483620002050884068e-01 1.08638116943747986727075716117952e-03
            9.95782584210465393503852737921989e-01 2.26470026775140012348028206723427e-04 3.99094576275946678195483130480170e-03 3.48362681281367705081334143479843e-06
            9.77911974785698800971545097127091e-01 1.18609971397786958559328329698701e-03 2.09019255003233296597020540730227e-02 4.19614023509517979179835045489000e-05
            9.46301233248777906581494789861608e-01 2.88355755060215558291392135004116e-03 5.08152092006199374019104197941488e-02 1.56824355141451987410636248476692e-04
            9.01852479486261682239955916884355e-01 5.27040081127802279709060329082604e-03 9.28771197024602923608682658596081e-02 3.78582930901072292964593124509065e-04
            8.45843521530176500888842383574229e-01 8.27801278054083981838839889633164e-03 1.45878465689282671435833549367089e-01 7.21943397762896685657119100198997e-04
            7.79885415536973658490182970126625e-01 1.18198817296222228462321268693813e-02 2.08294702733404113459414475073572e-01 1.18297693963182663605404609796778e-03
            7.05875580731421203850572965166066e-01 1.57941185861420613223415898573876e-02 2.78330300682436748704873252791003e-01 1.73855315600468388406718567296139e-03
            6.25943112845752769324292330566095e-01 2.00863935349841274402571400514717e-02 3.53970493619263082418768817660748e-01 2.34813280144964249898653640968860e-03
            5.42387506520867757053849800286116e-01 2.45732265497810610510587281396511e-02 4.33039266929351185364538423527847e-01 2.95773948543164282715123292177850e-03
            4.57612493479132464990755124745192e-01 2.91255402014410184896586741842839e-02 5.13261966319426554683502672560280e-01 3.50567558207354711130587787693003e-03
            3.74056887154247230675707669433905e-01 3.36123732162379659782480700869201e-02 5.92330739629514768651574740943033e-01 3.92934231554069573338905385639919e-03
            2.94124419268578907171729497349588e-01 3.79046481650800182183758124665474e-02 6.67970932566341102365470305812778e-01 4.17239147187788757958903929079497e-03
            2.20114584463026341509817029873375e-01 4.18788850215998653681026553385891e-02 7.38006530515373793122080314788036e-01 4.19139178980845478889039057435184e-03
            1.54156478469823443600006385167944e-01 4.54207539706812518653933352652530e-02 8.00422767559495262901236856123433e-01 3.96124218696890066648830242002077e-03
            9.81475205137383732711953143734718e-02 4.84283659399440749582232967895834e-02 8.53424113546317641976202139630914e-01 3.47870178622101088039553218322908e-03
            5.36987667512220934185052101383917e-02 5.08152092006199374019104197941488e-02 8.95486024048157913668433138809633e-01 2.76362176735508264721619120507512e-03
            2.20880252143010324950012090994278e-02 5.25126670372442344580932171993481e-02 9.25399307748454802435844612773508e-01 1.85777394944427109643725337662090e-03
            4.21741578953443996269356830453034e-03 5.34722967244469563063979933303926e-02 9.42310287486018527403075495385565e-01 8.22526182667714686296001058707361e-04
            9.95782584210465393503852737921989e-01 9.31543862984316882631941059145220e-05 4.12426140323617507893461109347299e-03 2.26608152699167430822171573345791e-06
            9.77911974785698800971545097127091e-01 4.87880857867600630471682787714371e-04 2.16001443564335995906056098192494e-02 2.72956788495251147545154341012008e-05
            9.46301233248777906581494789861608e-01 1.18609971397786351406111737816218e-03 5.25126670372442275191993132921198e-02 1.02013445545102223463787560753246e-04
            9.01852479486261682239955916884355e-01 2.16788490782858110292319331335875e-03 9.59796356059097327539930688544700e-02 2.46266271402467370760930309714354e-04
            8.45843521530176500888842383574229e-01 3.40501218338931438064931533915569e-03 1.50751466286434182562103956115607e-01 4.69620509058702004408225594289661e-04
            7.79885415536973658490182970126625e-01 4.86189649165471877295674474339648e-03 2.15252687971371620134775071164768e-01 7.69520483622539576244736547749881e-04
            7.05875580731421203850572965166066e-01 6.49662758894601569809035268576736e-03 2.87627791679632760502016708414885e-01 1.13091998718807778927242235766926e-03
            6.25943112845752769324292330566095e-01 8.26217795504596841382749516924378e-03 3.65794709199201262261880174264661e-01 1.52744844674981036618255370029829e-03
            5.42387506520867757053849800286116e-01 1.01077562943462390815385631981371e-02 4.47504737184786005599335112492554e-01 1.92399449474239380569740021087455e-03
            4.57612493479132464990755124745192e-01 1.19802689199547882092922179708694e-02 5.30407237600912795372209984634537e-01 2.28042413927405504189827034622340e-03
            3.74056887154247230675707669433905e-01 1.38258472592550640811737139301840e-02 6.12117265586497705243118616635911e-01 2.55601719498813181527530069558907e-03
            2.94124419268578907171729497349588e-01 1.55913976253550124601021664716427e-02 6.90284183106066095980679619970033e-01 2.71411943524554107737278840772888e-03
            2.20114584463026341509817029873375e-01 1.72261287226463111199592503908207e-02 7.62659286814327375125799335364718e-01 2.72647904544002410778813505487506e-03
            1.54156478469823443600006385167944e-01 1.86830130309117159459475487892632e-02 8.27160508499264812698470450413879e-01 2.57676789913674269758003809727143e-03
            9.81475205137383732711953143734718e-02 1.99201403064724544278440987454815e-02 8.81932339179789193117642298602732e-01 2.26287782223761387706550962661822e-03
            5.36987667512220934185052101383917e-02 2.09019255003231700651422642067701e-02 9.25399307748454691413542150257854e-01 1.79772190625012047437980555741888e-03
            2.20880252143010324950012090994278e-02 2.16001443564334365265988679993825e-02 9.56311830429265596897892010019859e-01 1.20847250706563940844673332719594e-03
            4.21741578953443996269356830453034e-03 2.19948708280026036798737720801000e-02 9.73787713382462949418538755708141e-01 5.35049099161351337031189423498745e-04
            9.95782584210465393503852737921989e-01 1.77865959418151070917362560619424e-05 4.19962919359279096553905219479930e-03 9.85298067232842572036330755502753e-07
            9.77911974785698800971545097127091e-01 9.31543862984284763142581176076362e-05 2.19948708280027702133274658535811e-02 1.18682312590707993870840428196622e-05
            9.46301233248777906581494789861608e-01 2.26470026775131067680105201311846e-04 5.34722967244469632452918972376210e-02 4.43557080935161589935189496713974e-05
            9.01852479486261682239955916884355e-01 4.13928902718295552848326579464810e-04 9.77335916110200209061531495535746e-02 1.07077207217523567720288135696194e-04
            8.45843521530176500888842383574229e-01 6.50141966357659279764080739738574e-04 1.53506336503465845577665049859206e-01 2.04192203324087532710293113602518e-04
            7.79885415536973658490182970126625e-01 9.28314724021179126997593833436895e-04 2.19186269739005151757638145681995e-01 3.34589482407512853016917642179351e-04
            7.05875580731421203850572965166066e-01 1.24044496991095173563102171954142e-03 2.92883974298667859592626427911455e-01 4.91726914631654432076557359465596e-04
            6.25943112845752769324292330566095e-01 1.57755342206842446607650831680303e-03 3.72479333732178785609789883892518e-01 6.64138507134092738980191139575027e-04
            5.42387506520867757053849800286116e-01 1.92994215548711800164993768191835e-03 4.55682551323645101959414205339272e-01 8.36557747131431842825544098474211e-04
            4.57612493479132464990755124745192e-01 2.28747363404732066000102364000668e-03 5.40100032886820180522136070067063e-01 9.91534271885047106820376683344875e-04
            3.74056887154247230675707669433905e-01 2.63986236746601527977662549062643e-03 6.23303250478286718916365316545125e-01 1.11136284023243404710246107214289e-03
            2.94124419268578907171729497349588e-01 2.97697081962348779338167759078715e-03 7.02898609911797644933528772526188e-01 1.18010610030287363923784393904270e-03
            2.20114584463026341509817029873375e-01 3.28910106551326061885553997399256e-03 7.76596314471460380524092670384562e-01 1.18548009055491126648163024981386e-03
            1.54156478469823443600006385167944e-01 3.56727382317678035766883581914044e-03 8.42276247706999714459641381836263e-01 1.12038529968405361526839758568030e-03
            9.81475205137383732711953143734718e-02 3.80348688681614484352611071926731e-03 8.98048992599445483620002050884068e-01 9.83905088178664002582229919369183e-04
            5.36987667512220934185052101383917e-02 3.99094576275930892211851741535611e-03 9.42310287486018638425377957901219e-01 7.81654101387012074757254520562810e-04
            2.20880252143010324950012090994278e-02 4.12426140323601201492786927360612e-03 9.73787713382463060440841218223795e-01 5.25446949429272231434084883261448e-04
            4.21741578953443996269356830453034e-03 4.19962919359262529944709640972178e-03 9.91582955016872880094069842016324e-01 2.32640722321324466262057462095925e-04
            ];
        
    case  34
        
        % ALG. DEG.:   34
        % PTS CARD.:  324
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.2e-15
        
        xyw_bar=[
            9.95782584210465393503852737921989e-01 4.19962919359279096553905219479930e-03 1.77865959418155306082098832121119e-05 9.85298067232849771816382417055635e-07
            9.77911974785698800971545097127091e-01 2.19948708280027112327292826421399e-02 9.31543862984877957256202307689819e-05 1.18682312590708857844446627582968e-05
            9.46301233248777906581494789861608e-01 5.34722967244469563063979933303926e-02 2.26470026775137112107216807999066e-04 4.43557080935164842541706953227276e-05
            9.01852479486261682239955916884355e-01 9.77335916110200625395165729969449e-02 4.13928902718255220527510118699865e-04 1.07077207217524340214336031618103e-04
            8.45843521530176500888842383574229e-01 1.53506336503465762310938202972466e-01 6.50141966357736800219413453305606e-04 2.04192203324089023488280281171114e-04
            7.79885415536973658490182970126625e-01 2.19186269739005068490911298795254e-01 9.28314724021273018905731078120880e-04 3.34589482407515238261697110289106e-04
            7.05875580731421203850572965166066e-01 2.92883974298667748570323965395801e-01 1.24044496991104757910306943813339e-03 4.91726914631658009943726561630228e-04
            6.25943112845752769324292330566095e-01 3.72479333732178730098638652634691e-01 1.57755342206850057706901679921430e-03 6.64138507134097509469750075794536e-04
            5.42387506520867757053849800286116e-01 4.55682551323644990937111742823618e-01 1.92994215548725200903845689026639e-03 8.36557747131437914357710017299041e-04
            4.57612493479132464990755124745192e-01 5.40100032886820069499833607551409e-01 2.28747363404746550941126770339906e-03 9.91534271885054262554715087674140e-04
            3.74056887154247230675707669433905e-01 6.23303250478286607894062854029471e-01 2.63986236746616143022947653662413e-03 1.11136284023244207019853746487570e-03
            2.94124419268578907171729497349588e-01 7.02898609911797533911226310010534e-01 2.97697081962355891704419263987802e-03 1.18010610030288231285522382307818e-03
            2.20114584463026341509817029873375e-01 7.76596314471460269501790207868908e-01 3.28910106551338898839276225771755e-03 1.18548009055491972325857563674845e-03
            1.54156478469823443600006385167944e-01 8.42276247706999603437338919320609e-01 3.56727382317689745150346425361931e-03 1.12038529968406163836447397841312e-03
            9.81475205137383732711953143734718e-02 8.98048992599445372597699588368414e-01 3.80348688681630964225632851594128e-03 9.83905088178671158316568323698448e-04
            5.36987667512220934185052101383917e-02 9.42310287486018416380773032869911e-01 3.99094576275949020072175699169748e-03 7.81654101387017712608551445185867e-04
            2.20880252143010324950012090994278e-02 9.73787713382462838396236293192487e-01 4.12426140323618461991372896591201e-03 5.25446949429276026141688582526967e-04
            4.21741578953443996269356830453034e-03 9.91582955016872658049464916985016e-01 4.19962919359284647669028345262632e-03 2.32640722321326146775424814627797e-04
            9.95782584210465393503852737921989e-01 4.12426140323617421157287310506945e-03 9.31543862984322845743889729419607e-05 2.26608152699165567349687613885045e-06
            9.77911974785698800971545097127091e-01 2.16001443564335406100074266078082e-02 4.87880857867658418447476265100704e-04 2.72956788495248945259491479831127e-05
            9.46301233248777906581494789861608e-01 5.25126670372442205803054093848914e-02 1.18609971397787283819980075350031e-03 1.02013445545101410312158196624921e-04
            9.01852479486261682239955916884355e-01 9.59796356059097743873564922978403e-02 2.16788490782854337268759081780445e-03 2.46266271402465364986911211531151e-04
            8.45843521530176500888842383574229e-01 1.50751466286434099295377109228866e-01 3.40501218338939981578050719690509e-03 4.69620509058698209700621895024142e-04
            7.79885415536973658490182970126625e-01 2.15252687971371536868048224278027e-01 4.86189649165480464176880559534766e-03 7.69520483622533396292353380374607e-04
            7.05875580731421203850572965166066e-01 2.87627791679632649479714245899231e-01 6.49662758894614666971278893470298e-03 1.13091998718806868197417347943201e-03
            6.25943112845752769324292330566095e-01 3.65794709199201206750728943006834e-01 8.26217795504602392497872642707080e-03 1.52744844674979800627778736554774e-03
            5.42387506520867757053849800286116e-01 4.47504737184785894577032649976900e-01 1.01077562943463483691175497369841e-02 1.92399449474237819318611641961070e-03
            4.57612493479132464990755124745192e-01 5.30407237600912684349907522118883e-01 1.19802689199548506593373531359248e-02 2.28042413927403682730177258974891e-03
            3.74056887154247230675707669433905e-01 6.12117265586497594220816154120257e-01 1.38258472592551751034761764458381e-02 2.55601719498811143227445796810571e-03
            2.94124419268578907171729497349588e-01 6.90284183106065984958377157454379e-01 1.55913976253551078698933451960329e-02 2.71411943524551895964846970343842e-03
            2.20114584463026341509817029873375e-01 7.62659286814327264103496872849064e-01 1.72261287226463943866860972775612e-02 2.72647904544000199006381635058460e-03
            1.54156478469823443600006385167944e-01 8.27160508499264590653865525382571e-01 1.86830130309119102349768581916578e-02 2.57676789913672144721745738138452e-03
            9.81475205137383732711953143734718e-02 8.81932339179789082095339836087078e-01 1.99201403064726001446160807972774e-02 2.26287782223759522878814287594196e-03
            5.36987667512220934185052101383917e-02 9.25399307748454580391239687742200e-01 2.09019255003233261902551021194085e-02 1.79772190625010594607069425165946e-03
            2.20880252143010324950012090994278e-02 9.56311830429265374853287084988551e-01 2.16001443564336481628629371698480e-02 1.20847250706562965062718095765604e-03
            4.21741578953443996269356830453034e-03 9.73787713382462727373933830676833e-01 2.19948708280027771522213697608095e-02 5.35049099161347000222499481481009e-04
            9.95782584210465393503852737921989e-01 3.99094576275946678195483130480170e-03 2.26470026775139714192430773209708e-04 3.48362681281366688641797438319436e-06
            9.77911974785698800971545097127091e-01 2.09019255003232741485508228151957e-02 1.18609971397792487990408005771314e-03 4.19614023509516759452390999296512e-05
            9.46301233248777906581494789861608e-01 5.08152092006199374019104197941488e-02 2.88355755060215601659479034424294e-03 1.56824355141451526624712942137307e-04
            9.01852479486261682239955916884355e-01 9.28771197024603478720194971174351e-02 5.27040081127796988802458599820966e-03 3.78582930901071208762420639004631e-04
            8.45843521530176500888842383574229e-01 1.45878465689282615924682318109262e-01 8.27801278054088318647529831650900e-03 7.21943397762894625672991377740573e-04
            7.79885415536973658490182970126625e-01 2.08294702733404057948263243815745e-01 1.18198817296222835615537860576296e-02 1.18297693963182316660709414435360e-03
            7.05875580731421203850572965166066e-01 2.78330300682436693193722021533176e-01 1.57941185861421029557050133007579e-02 1.73855315600467867989675774254010e-03
            6.25943112845752769324292330566095e-01 3.53970493619263082418768817660748e-01 2.00863935349841482569388517731568e-02 2.34813280144963599377350149666199e-03
            5.42387506520867757053849800286116e-01 4.33039266929351129853387192270020e-01 2.45732265497811130927630074438639e-02 2.95773948543163458721472203194480e-03
            4.57612493479132464990755124745192e-01 5.13261966319426554683502672560280e-01 2.91255402014409803257422026945278e-02 3.50567558207353713664589101028923e-03
            3.74056887154247230675707669433905e-01 5.92330739629514768651574740943033e-01 3.36123732162380006727175896230619e-02 3.92934231554068445768646000715307e-03
            2.94124419268578907171729497349588e-01 6.67970932566341102365470305812778e-01 3.79046481650799904628001968376338e-02 4.17239147187787543652470745314531e-03
            2.20114584463026341509817029873375e-01 7.38006530515373793122080314788036e-01 4.18788850215998653681026553385891e-02 4.19139178980844264582605873670218e-03
            1.54156478469823443600006385167944e-01 8.00422767559495262901236856123433e-01 4.54207539706812379876055274507962e-02 3.96124218696888939078570857077466e-03
            9.81475205137383732711953143734718e-02 8.53424113546317641976202139630914e-01 4.84283659399440402637537772534415e-02 3.47870178622100090573554531658829e-03
            5.36987667512220934185052101383917e-02 8.95486024048157913668433138809633e-01 5.08152092006199929130616510519758e-02 2.76362176735507484096054930944319e-03
            2.20880252143010324950012090994278e-02 9.25399307748454802435844612773508e-01 5.25126670372442205803054093848914e-02 1.85777394944426567542639094909873e-03
            4.21741578953443996269356830453034e-03 9.42310287486018527403075495385565e-01 5.34722967244469771230797050520778e-02 8.22526182667712301051221590597606e-04
            9.95782584210465393503852737921989e-01 3.80348688681629489710678271308097e-03 4.13928902718311599040479364930434e-04 4.60112595870666033420282384214239e-06
            9.77911974785698800971545097127091e-01 1.99201403064725515723587534466787e-02 2.16788490782864745609614942623011e-03 5.54220380066369058889678977930515e-05
            9.46301233248777906581494789861608e-01 4.84283659399440749582232967895834e-02 5.27040081127801846028191334880830e-03 2.07131432317793535719668840400232e-04
            9.01852479486261682239955916884355e-01 8.85145847307436811624015149391198e-02 9.63293578299463659764256817652495e-03 5.00027082259496850544233659263682e-04
            8.45843521530176500888842383574229e-01 1.39026402336880794408102701709140e-01 1.51300761329427047030549147166312e-02 9.53532822731037151345323543694121e-04
            7.79885415536973658490182970126625e-01 1.98510883769068419146819337584020e-01 2.16037006939579223629976922893547e-02 1.56245952794670977040181192307955e-03
            7.05875580731421203850572965166066e-01 2.65256836794824535363801487619639e-01 2.88675824737542607856255472142948e-02 2.29625688585845255662576391841867e-03
            6.25943112845752769324292330566095e-01 3.37344131148970660660779685713351e-01 3.67127560052765700149279837205540e-02 3.10138122358584586088503520784343e-03
            5.42387506520867757053849800286116e-01 4.12698961888046111923245007346850e-01 4.49135315910861310229051923670340e-02 3.90654127343780140091400454593895e-03
            4.57612493479132464990755124745192e-01 4.89153517598215292760954753248370e-01 5.32339889226522422482901220064377e-02 4.63024766721629923299508035938743e-03
            3.74056887154247230675707669433905e-01 5.64508348337291021579176231171004e-01 6.14347645084617477451160993950907e-02 5.18982081036296546971975374162867e-03
            2.94124419268578907171729497349588e-01 6.36595642691436980342700735491235e-01 6.92799380399841124855697671591770e-02 5.51083676372266829945889554664973e-03
            2.20114584463026341509817029873375e-01 7.03341595717193235337560963671422e-01 7.65438198197804231526220064552035e-02 5.53593212001411878742107575135378e-03
            1.54156478469823443600006385167944e-01 7.62826077149380887831853215175215e-01 8.30174443807956130569891683990136e-02 5.23195371793154853645724244870507e-03
            9.81475205137383732711953143734718e-02 8.13337894755518098222069056646433e-01 8.85145847307435840178868602379225e-02 4.59462105191828489719751971165351e-03
            5.36987667512220934185052101383917e-02 8.53424113546317641976202139630914e-01 9.28771197024602646052926502306946e-02 3.65015328480432048319226367993906e-03
            2.20880252143010324950012090994278e-02 8.81932339179789304139944761118386e-01 9.59796356059097188762052610400133e-02 2.45372205563346460974183393943804e-03
            4.21741578953443996269356830453034e-03 8.98048992599445483620002050884068e-01 9.77335916110200209061531495535746e-02 1.08638116943748984193074402782031e-03
            9.95782584210465393503852737921989e-01 3.56727382317692130395125893471686e-03 6.50141966357685192196003143294547e-04 5.58629407564891192932484548228089e-06
            9.77911974785698800971545097127091e-01 1.86830130309118096210152515368463e-02 3.40501218338938940743965133606252e-03 6.72887039727753337578150527598098e-05
            9.46301233248777906581494789861608e-01 4.54207539706812518653933352652530e-02 8.27801278054084155311187487313873e-03 2.51481290367198343903981649916091e-04
            9.01852479486261682239955916884355e-01 8.30174443807957102015038231002109e-02 1.51300761329426075585402600154339e-02 6.07090167137135420204197000515478e-04
            8.45843521530176500888842383574229e-01 1.30392258615606299043676585824869e-01 2.37642198542172000674810306009022e-02 1.15769809528458856966171008906485e-03
            7.79885415536973658490182970126625e-01 1.86182495262357639020933675055858e-01 3.39320892006687024888833548175171e-02 1.89700488157541606072664652771209e-03
            7.05875580731421203850572965166066e-01 2.48783234562152733015949479522533e-01 4.53411847064260631334775553114014e-02 2.78791894696245583923355226829699e-03
            6.25943112845752769324292330566095e-01 3.16393594683164303127398397919023e-01 5.76632924710829275483092715148814e-02 3.76543213794485387077570770486545e-03
            5.42387506520867757053849800286116e-01 3.87068562980594110811694008589257e-01 7.05439304985381321344561911246274e-02 4.74298868753809162862600246057809e-03
            4.57612493479132464990755124745192e-01 4.58774958549582168032543449953664e-01 8.36125479712853669767014253011439e-02 5.62165116632197363333300899057576e-03
            3.74056887154247230675707669433905e-01 5.29449926847012197761443985655205e-01 9.64931859987405715628483449108899e-02 6.30103707370781307428586615060340e-03
            2.94124419268578907171729497349588e-01 5.97060286968023601339439210278215e-01 1.08815293763397491488831292372197e-01 6.69078722063623423238487575304134e-03
            2.20114584463026341509817029873375e-01 6.59661026267818861867908708518371e-01 1.20224389269154796622274261608254e-01 6.72125585840781074320471333294336e-03
            1.54156478469823443600006385167944e-01 7.15451262914570174089590182120446e-01 1.30392258615606326799252201453783e-01 6.35219125076198765089685593920876e-03
            9.81475205137383732711953143734718e-02 7.62826077149380887831853215175215e-01 1.39026402336880794408102701709140e-01 5.57839637352541654891080469269582e-03
            5.36987667512220934185052101383917e-02 8.00422767559495262901236856123433e-01 1.45878465689282643680257933738176e-01 4.43170429436461066341479408947635e-03
            2.20880252143010324950012090994278e-02 8.27160508499264812698470450413879e-01 1.50751466286434210317679571744520e-01 2.97909970422267694409312888126351e-03
            4.21741578953443996269356830453034e-03 8.42276247706999714459641381836263e-01 1.53506336503465790066513818601379e-01 1.31899120893249569702598833487173e-03
            9.95782584210465393503852737921989e-01 3.28910106551339028943536924032287e-03 9.28314724021216206711892837688538e-04 6.41076541820288155643656549664833e-06
            9.77911974785698800971545097127091e-01 1.72261287226463978561330492311754e-02 4.86189649165480117232185364173347e-03 7.72197257471179948759218425458073e-05
            9.46301233248777906581494789861608e-01 4.18788850215998653681026553385891e-02 1.18198817296222280504025547998026e-02 2.88596972837273804766283769041024e-04
            9.01852479486261682239955916884355e-01 7.65438198197804925415610455274873e-02 2.16037006939578252184830375881575e-02 6.96689539882782587522369599497551e-04
            8.45843521530176500888842383574229e-01 1.20224389269154782744486453793797e-01 3.39320892006687163666711626319739e-02 1.32856072621054515509042559529007e-03
            7.79885415536973658490182970126625e-01 1.71664154169695548635132809067727e-01 4.84504302933307928746842208056478e-02 2.17698050411946168736232465334979e-03
            7.05875580731421203850572965166066e-01 2.29383344940846584103510963359440e-01 6.47410743277322120459160714744939e-02 3.19938301348079122410350549898794e-03
            6.25943112845752769324292330566095e-01 2.91721510872756939480865412406274e-01 8.23353762814902911948422570276307e-02 4.32116566146270326270739658980347e-03
            5.42387506520867757053849800286116e-01 3.56885309631883640424376835653675e-01 1.00727183847248602521773364060209e-01 5.44299806727676499518153718781832e-03
            4.57612493479132464990755124745192e-01 4.23000105905089796021201209441642e-01 1.19387400615777738988043665813166e-01 6.45134079986121270616772349626444e-03
            3.74056887154247230675707669433905e-01 4.88163904664216719009317557720351e-01 1.37779208181536050314974772845744e-01 7.23099608146713073580924557859362e-03
            2.94124419268578907171729497349588e-01 5.50502070596126880097642697364790e-01 1.55373510135294212730627805285621e-01 7.67826876883956777747330946226612e-03
            2.20114584463026341509817029873375e-01 6.08221261367278054343898929801071e-01 1.71664154169695604146284040325554e-01 7.71323421940852804040700263499275e-03
            1.54156478469823443600006385167944e-01 6.59661026267818861867908708518371e-01 1.86182495262357639020933675055858e-01 7.28970001377263459446664839447294e-03
            9.81475205137383732711953143734718e-02 7.03341595717193235337560963671422e-01 1.98510883769068446902394953212934e-01 6.40170210807796801616431636716698e-03
            5.36987667512220934185052101383917e-02 7.38006530515373793122080314788036e-01 2.08294702733404113459414475073572e-01 5.08577175660299141135567424498731e-03
            2.20880252143010324950012090994278e-02 7.62659286814327375125799335364718e-01 2.15252687971371647890350686793681e-01 3.41877980331543570094443573736953e-03
            4.21741578953443996269356830453034e-03 7.76596314471460380524092670384562e-01 2.19186269739005124002062530053081e-01 1.51365880754421680033650066121709e-03
            9.95782584210465393503852737921989e-01 2.97697081962360575457804401366957e-03 1.24044496991100074156921806434184e-03 7.05081495465016400267434931725674e-06
            9.77911974785698800971545097127091e-01 1.55913976253550939921055373815761e-02 6.49662758894610503634936549133272e-03 8.49293277126948574365719601964031e-05
            9.46301233248777906581494789861608e-01 3.79046481650800251572697163737757e-02 1.57941185861420682612354937646160e-02 3.17410436852054054709948438528500e-04
            9.01852479486261682239955916884355e-01 6.92799380399841541189331906025473e-02 2.88675824737541636411108925130975e-02 7.66246884124938457648334999561257e-04
            8.45843521530176500888842383574229e-01 1.08815293763397449855467868928827e-01 4.53411847064260492556897474969446e-02 1.46120396324720780260875141465249e-03
            7.79885415536973658490182970126625e-01 1.55373510135294129463900958398881e-01 6.47410743277322120459160714744939e-02 2.39432980199893920975062755474028e-03
            7.05875580731421203850572965166066e-01 2.07615245258500030756110277252446e-01 8.65091740100787653933167575814878e-02 3.51880877329431053512909244318507e-03
            6.25943112845752769324292330566095e-01 2.64037622446591957459105515226838e-01 1.10019264707655273216602154207067e-01 4.75258997636244292367413422084610e-03
            5.42387506520867757053849800286116e-01 3.23017484584536107572461105519324e-01 1.34595008894596135373689094194560e-01 5.98642590507472878436079355424226e-03
            4.57612493479132464990755124745192e-01 3.82858096146884874233506934615434e-01 1.59529410373982660775737940639374e-01 7.09544137429342098438711516905641e-03
            3.74056887154247230675707669433905e-01 4.41837958284829246391467449939228e-01 1.84105154560923522932824880626868e-01 7.95293728319223895695255777127386e-03
            2.94124419268578907171729497349588e-01 4.98260335472921034316584609769052e-01 2.07615245258500058511685892881360e-01 8.44486558616486596151595733772410e-03
            2.20114584463026341509817029873375e-01 5.50502070596126991119945159880444e-01 2.29383344940846667370237810246181e-01 8.48332198032143366961399522097054e-03
            1.54156478469823443600006385167944e-01 5.97060286968023712361741672793869e-01 2.48783234562152788527100710780360e-01 8.01750220435144302089103263142533e-03
            9.81475205137383732711953143734718e-02 6.36595642691437091365003198006889e-01 2.65256836794824590874952718877466e-01 7.04084676545610639586403323164632e-03
            5.36987667512220934185052101383917e-02 6.67970932566341213387772768328432e-01 2.78330300682436693193722021533176e-01 5.59353419103050721744585516148618e-03
            2.20880252143010324950012090994278e-02 6.90284183106066207002982082485687e-01 2.87627791679632816013167939672712e-01 3.76011009472091784280878279389526e-03
            4.21741578953443996269356830453034e-03 7.02898609911797644933528772526188e-01 2.92883974298667859592626427911455e-01 1.66478219997987539162387360391904e-03
            9.95782584210465393503852737921989e-01 2.63986236746611979686605309325387e-03 1.57755342206848669928120898475754e-03 7.48802741672358854359281554735617e-06
            9.77911974785698800971545097127091e-01 1.38258472592551334701127530024678e-02 8.26217795504606555834214987044106e-03 9.01956920564390648466235322899820e-05
            9.46301233248777906581494789861608e-01 3.36123732162379659782480700869201e-02 2.00863935349841274402571400514717e-02 3.37092672093862813238046571839845e-04
            9.01852479486261682239955916884355e-01 6.14347645084617755006917150240042e-02 3.67127560052765422593523680916405e-02 8.13760921710537823807463109204718e-04
            8.45843521530176500888842383574229e-01 9.64931859987405576850605370964331e-02 5.76632924710829414260970793293382e-02 1.55181144429327008867069981334907e-03
            7.79885415536973658490182970126625e-01 1.37779208181536050314974772845744e-01 8.23353762814902911948422570276307e-02 2.54279928169466205509219314251368e-03
            7.05875580731421203850572965166066e-01 1.84105154560923495177249264997954e-01 1.10019264707655300972177769835980e-01 3.73700582671760610578570727113856e-03
            6.25943112845752769324292330566095e-01 2.34138332326721987364237520523602e-01 1.39918554827525243311470148910303e-01 5.04729230200218811214218916916252e-03
            5.42387506520867757053849800286116e-01 2.86439388645434722668881022400456e-01 1.71173104833697520277269177313428e-01 6.35763689640157607374515436049478e-03
            4.57612493479132464990755124745192e-01 3.39503724200317880121957614392159e-01 2.02883782320549654887287260862649e-01 7.53542106638652956634993884676987e-03
            3.74056887154247230675707669433905e-01 3.91804780519030781960054810042493e-01 2.34138332326721987364237520523602e-01 8.44608925394517602580624071606508e-03
            2.94124419268578907171729497349588e-01 4.41837958284829190880316218681401e-01 2.64037622446591901947954283969011e-01 8.96852143283705625920187998190158e-03
            2.20114584463026341509817029873375e-01 4.88163904664216719009317557720351e-01 2.91721510872756939480865412406274e-01 9.00936246123516977502010405487454e-03
            1.54156478469823443600006385167944e-01 5.29449926847012197761443985655205e-01 3.16393594683164303127398397919023e-01 8.51465776735933939367573231038477e-03
            9.81475205137383732711953143734718e-02 5.64508348337291021579176231171004e-01 3.37344131148970660660779685713351e-01 7.47744111223818102435201282673916e-03
            5.36987667512220934185052101383917e-02 5.92330739629514768651574740943033e-01 3.53970493619263137929920048918575e-01 5.94038244489645574086811308234246e-03
            2.20880252143010324950012090994278e-02 6.12117265586497705243118616635911e-01 3.65794709199201317773031405522488e-01 3.99326994968149037723570415892027e-03
            4.21741578953443996269356830453034e-03 6.23303250478286718916365316545125e-01 3.72479333732178785609789883892518e-01 1.76801331994979769521303403223556e-03
            9.95782584210465393503852737921989e-01 2.28747363404741216666438141658091e-03 1.92994215548719432948288066143050e-03 7.70982414572235148547598576618256e-06
            9.77911974785698800971545097127091e-01 1.19802689199548523940608291127319e-02 1.01077562943463466343940737601770e-02 9.28673047996321385190096608930332e-05
            9.46301233248777906581494789861608e-01 2.91255402014410288979995300451264e-02 2.45732265497810645205056800932653e-02 3.47077418126294544656329499332514e-04
            9.01852479486261682239955916884355e-01 5.32339889226522769427596415425796e-02 4.49135315910860408172844415730651e-02 8.37864667674303119880574453048894e-04
            8.45843521530176500888842383574229e-01 8.36125479712853669767014253011439e-02 7.05439304985381321344561911246274e-02 1.59777638047905296524164864280237e-03
            7.79885415536973658490182970126625e-01 1.19387400615777766743619281442079e-01 1.00727183847248574766197748431296e-01 2.61811745720249861718587069958630e-03
            7.05875580731421203850572965166066e-01 1.59529410373982688531313556268287e-01 1.34595008894596107618113478565647e-01 3.84769661649277854367778850530613e-03
            6.25943112845752769324292330566095e-01 2.02883782320549765909589723378303e-01 1.71173104833697464766117946055601e-01 5.19679401461404545659039655447486e-03
            5.42387506520867757053849800286116e-01 2.48203299290943368848871841692016e-01 2.09409194188188874097278358021867e-01 6.54595125334883100320926274662270e-03
            4.57612493479132464990755124745192e-01 2.94184207229924221671524264820619e-01 2.48203299290943313337720610434189e-01 7.75862160387657488064272470751348e-03
            3.74056887154247230675707669433905e-01 3.39503724200317991144260076907813e-01 2.86439388645434778180032253658283e-01 8.69626394817410094950638210775651e-03
            2.94124419268578907171729497349588e-01 3.82858096146884985255809397131088e-01 3.23017484584536107572461105519324e-01 9.23417066287538987334748696866882e-03
            2.20114584463026341509817029873375e-01 4.23000105905089962554654903215123e-01 3.56885309631883695935528066911502e-01 9.27622140993551204390943354383126e-03
            1.54156478469823443600006385167944e-01 4.58774958549582334565997143727145e-01 3.87068562980594166322845239847084e-01 8.76686347338098345693957469393354e-03
            9.81475205137383732711953143734718e-02 4.89153517598215514805559678279678e-01 4.12698961888046167434396238604677e-01 7.69892427297972671562664714883795e-03
            5.36987667512220934185052101383917e-02 5.13261966319426776728107597591588e-01 4.33039266929351129853387192270020e-01 6.11633764937889387591773981966981e-03
            2.20880252143010324950012090994278e-02 5.30407237600913017416814909665845e-01 4.47504737184786005599335112492554e-01 4.11155132921681588237250437600778e-03
            4.21741578953443996269356830453034e-03 5.40100032886820402566740995098371e-01 4.55682551323645101959414205339272e-01 1.82038219487075641350726229461543e-03
            9.95782584210465393503852737921989e-01 1.92994215548719541368505314693493e-03 2.28747363404741086562177443397559e-03 7.70982414572235656767366929198460e-06
            9.77911974785698800971545097127091e-01 1.01077562943462963274132704327712e-02 1.19802689199549027010416324401376e-02 9.28673047996322062816454412370604e-05
            9.46301233248777906581494789861608e-01 2.45732265497810749288465359541078e-02 2.91255402014410184896586741842839e-02 3.47077418126294869916981244983845e-04
            9.01852479486261682239955916884355e-01 4.49135315910861240840112884598057e-02 5.32339889226521936760327946558391e-02 8.37864667674303770401877944351554e-04
            8.45843521530176500888842383574229e-01 7.05439304985381043788805754957139e-02 8.36125479712853947322770409300574e-02 1.59777638047905426628425562540770e-03
            7.79885415536973658490182970126625e-01 1.00727183847248574766197748431296e-01 1.19387400615777766743619281442079e-01 2.61811745720250078559021567059517e-03
            7.05875580731421203850572965166066e-01 1.34595008894596107618113478565647e-01 1.59529410373982688531313556268287e-01 3.84769661649278201312474045892031e-03
            6.25943112845752769324292330566095e-01 1.71173104833697548032844792942342e-01 2.02883782320549682642862876491563e-01 5.19679401461404979339908649649260e-03
            5.42387506520867757053849800286116e-01 2.09409194188188929608429589279694e-01 2.48203299290943313337720610434189e-01 6.54595125334883620737969067704398e-03
            4.57612493479132464990755124745192e-01 2.48203299290943424360023072949843e-01 2.94184207229924110649221802304965e-01 7.75862160387658095217489062633831e-03
            3.74056887154247230675707669433905e-01 2.86439388645434889202334716173937e-01 3.39503724200317880121957614392159e-01 8.69626394817410962312376199179198e-03
            2.94124419268578907171729497349588e-01 3.23017484584536274105914799292805e-01 3.82858096146884818722355703357607e-01 9.23417066287539681224139087589720e-03
            2.20114584463026341509817029873375e-01 3.56885309631883862468981760684983e-01 4.23000105905089796021201209441642e-01 9.27622140993551898280333745105963e-03
            1.54156478469823443600006385167944e-01 3.87068562980594332856298933620565e-01 4.58774958549582168032543449953664e-01 8.76686347338099039583347860116191e-03
            9.81475205137383732711953143734718e-02 4.12698961888046389479001163635985e-01 4.89153517598215292760954753248370e-01 7.69892427297973278715881306766278e-03
            5.36987667512220934185052101383917e-02 4.33039266929351407409143348559155e-01 5.13261966319426443661200210044626e-01 6.11633764937889908008816775009109e-03
            2.20880252143010324950012090994278e-02 4.47504737184786227643940037523862e-01 5.30407237600912795372209984634537e-01 4.11155132921681935181945632962197e-03
            4.21741578953443996269356830453034e-03 4.55682551323645324004019130370580e-01 5.40100032886820180522136070067063e-01 1.82038219487075793139030377432164e-03
            9.95782584210465393503852737921989e-01 1.57755342206848669928120898475754e-03 2.63986236746611979686605309325387e-03 7.48802741672362666007544199087143e-06
            9.77911974785698800971545097127091e-01 8.26217795504601004719091861261404e-03 1.38258472592551889812639842602948e-02 9.01956920564395256325468386293664e-05
            9.46301233248777906581494789861608e-01 2.00863935349841274402571400514717e-02 3.36123732162379659782480700869201e-02 3.37092672093864493751413924371718e-04
            9.01852479486261682239955916884355e-01 3.67127560052765977705035993494675e-02 6.14347645084617199895404837661772e-02 8.13760921710541943775718554121568e-04
            8.45843521530176500888842383574229e-01 5.76632924710828928538397519787395e-02 9.64931859987405993184239605398034e-02 1.55181144429327811176677620608189e-03
            7.79885415536973658490182970126625e-01 8.23353762814902495614788335842604e-02 1.37779208181536105826126004103571e-01 2.54279928169467506551826296856689e-03
            7.05875580731421203850572965166066e-01 1.10019264707655259338814346392610e-01 1.84105154560923522932824880626868e-01 3.73700582671762518774394301601660e-03
            6.25943112845752769324292330566095e-01 1.39918554827525243311470148910303e-01 2.34138332326721987364237520523602e-01 5.04729230200221413299432882126894e-03
            5.42387506520867757053849800286116e-01 1.71173104833697437010542330426688e-01 2.86439388645434833691183484916110e-01 6.35763689640160903349119791982957e-03
            4.57612493479132464990755124745192e-01 2.02883782320549682642862876491563e-01 3.39503724200317824610806383134332e-01 7.53542106638656773026641033652595e-03
            3.74056887154247230675707669433905e-01 2.34138332326721987364237520523602e-01 3.91804780519030781960054810042493e-01 8.44608925394521939389314013624244e-03
            2.94124419268578907171729497349588e-01 2.64037622446591901947954283969011e-01 4.41837958284829190880316218681401e-01 8.96852143283710136201225537888604e-03
            2.20114584463026341509817029873375e-01 2.91721510872756939480865412406274e-01 4.88163904664216719009317557720351e-01 9.00936246123521487783047945185899e-03
            1.54156478469823443600006385167944e-01 3.16393594683164303127398397919023e-01 5.29449926847012197761443985655205e-01 8.51465776735938276176263173056213e-03
            9.81475205137383732711953143734718e-02 3.37344131148970660660779685713351e-01 5.64508348337291021579176231171004e-01 7.47744111223821832090674632809169e-03
            5.36987667512220934185052101383917e-02 3.53970493619263082418768817660748e-01 5.92330739629514768651574740943033e-01 5.94038244489648609852894267646661e-03
            2.20880252143010324950012090994278e-02 3.65794709199201262261880174264661e-01 6.12117265586497705243118616635911e-01 3.99326994968151119391741588060540e-03
            4.21741578953443996269356830453034e-03 3.72479333732178785609789883892518e-01 6.23303250478286718916365316545125e-01 1.76801331994980658567084841337191e-03
            9.95782584210465393503852737921989e-01 1.24044496991100139209052155564450e-03 2.97697081962360488721630602526602e-03 7.05081495465018179036624165756386e-06
            9.77911974785698800971545097127091e-01 6.49662758894605212728334819871634e-03 1.55913976253551460338098166857890e-02 8.49293277126950742770064572972899e-05
            9.46301233248777906581494789861608e-01 1.57941185861420717306824457182302e-02 3.79046481650800182183758124665474e-02 3.17410436852054867861577802656825e-04
            9.01852479486261682239955916884355e-01 2.88675824737542330300499315853813e-02 6.92799380399840847299941515302635e-02 7.66246884124940409212245473469238e-04
            8.45843521530176500888842383574229e-01 4.53411847064260076223263240535744e-02 1.08815293763397491488831292372197e-01 1.46120396324721148889613786536756e-03
            7.79885415536973658490182970126625e-01 6.47410743277321704125526480311237e-02 1.55373510135294157219476574027794e-01 2.39432980199894528128279347356511e-03
            7.05875580731421203850572965166066e-01 8.65091740100787376377411419525743e-02 2.07615245258500058511685892881360e-01 3.51880877329431964242734132142232e-03
            6.25943112845752769324292330566095e-01 1.10019264707655314849965577650437e-01 2.64037622446591901947954283969011e-01 4.75258997636245419937672807009221e-03
            5.42387506520867757053849800286116e-01 1.34595008894596107618113478565647e-01 3.23017484584536163083612336777151e-01 5.98642590507474352951033935710257e-03
            4.57612493479132464990755124745192e-01 1.59529410373982716286889171897201e-01 3.82858096146884818722355703357607e-01 7.09544137429343833162187493712736e-03
            3.74056887154247230675707669433905e-01 1.84105154560923578443976111884695e-01 4.41837958284829190880316218681401e-01 7.95293728319225977363426949295899e-03
            2.94124419268578907171729497349588e-01 2.07615245258500114022837124139187e-01 4.98260335472920978805433378511225e-01 8.44486558616488677819766905940924e-03
            2.20114584463026341509817029873375e-01 2.29383344940846722881389041504008e-01 5.50502070596126991119945159880444e-01 8.48332198032145448629570694265567e-03
            1.54156478469823443600006385167944e-01 2.48783234562152871793827557667100e-01 5.97060286968023601339439210278215e-01 8.01750220435146210284926837630337e-03
            9.81475205137383732711953143734718e-02 2.65256836794824701897255181393120e-01 6.36595642691436980342700735491235e-01 7.04084676545612374309879299971726e-03
            5.36987667512220934185052101383917e-02 2.78330300682436859727175715306657e-01 6.67970932566340991343167843297124e-01 5.59353419103052109523366297594293e-03
            2.20880252143010324950012090994278e-02 2.87627791679632927035470402188366e-01 6.90284183106066095980679619970033e-01 3.76011009472092738378790066633428e-03
            4.21741578953443996269356830453034e-03 2.92883974298667915103777659169282e-01 7.02898609911797533911226310010534e-01 1.66478219997987972843256354593677e-03
            9.95782584210465393503852737921989e-01 9.28314724021216098291675589138094e-04 3.28910106551339028943536924032287e-03 6.41076541820290527335908861705782e-06
            9.77911974785698800971545097127091e-01 4.86189649165474479380888439550290e-03 1.72261287226464533672842804890024e-02 7.72197257471182794789921199907212e-05
            9.46301233248777906581494789861608e-01 1.18198817296222263156790788229955e-02 4.18788850215998653681026553385891e-02 2.88596972837274888968456254545458e-04
            9.01852479486261682239955916884355e-01 2.16037006939578772601873168923703e-02 7.65438198197804370304098142696603e-02 6.96689539882785189607583564708193e-04
            8.45843521530176500888842383574229e-01 3.39320892006686539166260274669185e-02 1.20224389269154852133425492866081e-01 1.32856072621055014242041902861047e-03
            7.79885415536973658490182970126625e-01 4.84504302933307443024268934550491e-02 1.71664154169695604146284040325554e-01 2.17698050411946949361796654898171e-03
            7.05875580731421203850572965166066e-01 6.47410743277321426569770324022102e-02 2.29383344940846667370237810246181e-01 3.19938301348080293348696834243583e-03
            6.25943112845752769324292330566095e-01 8.23353762814902634392666413987172e-02 2.91721510872756994992016643664101e-01 4.32116566146271974258041836947086e-03
            5.42387506520867757053849800286116e-01 1.00727183847248533132834324987925e-01 3.56885309631883695935528066911502e-01 5.44299806727678581186324890950345e-03
            4.57612493479132464990755124745192e-01 1.19387400615777752865831473627622e-01 4.23000105905089796021201209441642e-01 6.45134079986123699229638717156377e-03
            3.74056887154247230675707669433905e-01 1.37779208181536078070550388474658e-01 4.88163904664216663498166326462524e-01 7.23099608146715762402312321910358e-03
            2.94124419268578907171729497349588e-01 1.55373510135294157219476574027794e-01 5.50502070596126991119945159880444e-01 7.67826876883959640041066307958317e-03
            2.20114584463026341509817029873375e-01 1.71664154169695576390708424696641e-01 6.08221261367278054343898929801071e-01 7.71323421940855666334435625230981e-03
            1.54156478469823443600006385167944e-01 1.86182495262357666776509290684771e-01 6.59661026267818861867908708518371e-01 7.28970001377266148268052603498290e-03
            9.81475205137383732711953143734718e-02 1.98510883769068474657970568841847e-01 7.03341595717193235337560963671422e-01 6.40170210807799230229298004246630e-03
            5.36987667512220934185052101383917e-02 2.08294702733404113459414475073572e-01 7.38006530515373793122080314788036e-01 5.08577175660301049331390998986535e-03
            2.20880252143010324950012090994278e-02 2.15252687971371620134775071164768e-01 7.62659286814327375125799335364718e-01 3.41877980331544871137050556342274e-03
            4.21741578953443996269356830453034e-03 2.19186269739005151757638145681995e-01 7.76596314471460380524092670384562e-01 1.51365880754422265502823208294103e-03
            9.95782584210465393503852737921989e-01 6.50141966357684975355568646193660e-04 3.56727382317692173763212792891864e-03 5.58629407564889583569884765057445e-06
            9.77911974785698800971545097127091e-01 3.40501218338933172788407510722664e-03 1.86830130309118686016134347482875e-02 6.72887039727751304699077117277284e-05
            9.46301233248777906581494789861608e-01 8.27801278054083981838839889633164e-03 4.54207539706812518653933352652530e-02 2.51481290367197639172569534338209e-04
            9.01852479486261682239955916884355e-01 1.51300761329426648044149672500680e-02 8.30174443807956546903525918423838e-02 6.07090167137133577060503775157940e-04
            8.45843521530176500888842383574229e-01 2.37642198542171376174358954358468e-02 1.30392258615606354554827817082696e-01 1.15769809528458510021475813545067e-03
            7.79885415536973658490182970126625e-01 3.39320892006686469777321235596901e-02 1.86182495262357694532084906313685e-01 1.89700488157541042287534960308903e-03
            7.05875580731421203850572965166066e-01 4.53411847064259798667507084246608e-02 2.48783234562152816282676326409273e-01 2.78791894696244759929704137846329e-03
            6.25943112845752769324292330566095e-01 5.76632924710828928538397519787395e-02 3.16393594683164358638549629176850e-01 3.76543213794484259507311385561934e-03
            5.42387506520867757053849800286116e-01 7.05439304985380627455171520523436e-02 3.87068562980594166322845239847084e-01 4.74298868753807775083819464612134e-03
            4.57612493479132464990755124745192e-01 8.36125479712853392211258096722304e-02 4.58774958549582168032543449953664e-01 5.62165116632195715345998721090837e-03
            3.74056887154247230675707669433905e-01 9.64931859987405576850605370964331e-02 5.29449926847012197761443985655205e-01 6.30103707370779485968936839412891e-03
            2.94124419268578907171729497349588e-01 1.08815293763397435977680061114370e-01 5.97060286968023712361741672793869e-01 6.69078722063621428306490201975976e-03
            2.20114584463026341509817029873375e-01 1.20224389269154782744486453793797e-01 6.59661026267818861867908708518371e-01 6.72125585840779079388473959966177e-03
            1.54156478469823443600006385167944e-01 1.30392258615606299043676585824869e-01 7.15451262914570174089590182120446e-01 6.35219125076196856893862019433072e-03
            9.81475205137383732711953143734718e-02 1.39026402336880794408102701709140e-01 7.62826077149380887831853215175215e-01 5.57839637352540006903778291302842e-03
            5.36987667512220934185052101383917e-02 1.45878465689282615924682318109262e-01 8.00422767559495262901236856123433e-01 4.43170429436459765298872426342314e-03
            2.20880252143010324950012090994278e-02 1.50751466286434127050952724857780e-01 8.27160508499264923720772912929533e-01 2.97909970422266827047574899722804e-03
            4.21741578953443996269356830453034e-03 1.53506336503465762310938202972466e-01 8.42276247706999714459641381836263e-01 1.31899120893249179389816738705576e-03
            9.95782584210465393503852737921989e-01 4.13928902718311924301131110581764e-04 3.80348688681629446342591371887920e-03 4.60112595870661882958840838142578e-06
            9.77911974785698800971545097127091e-01 2.16788490782859194494491816840309e-03 1.99201403064726070835099847045058e-02 5.54220380066363976691995452128481e-05
            9.46301233248777906581494789861608e-01 5.27040081127802279709060329082604e-03 4.84283659399440680193293928823550e-02 2.07131432317791638365866990767472e-04
            9.01852479486261682239955916884355e-01 9.63293578299469557824075138796616e-03 8.85145847307436256512502836812928e-02 5.00027082259492296895109220145059e-04
            8.45843521530176500888842383574229e-01 1.51300761329426648044149672500680e-02 1.39026402336880822163678317338054e-01 9.53532822731028477727943659658649e-04
            7.79885415536973658490182970126625e-01 2.16037006939578703212934129851419e-02 1.98510883769068474657970568841847e-01 1.56245952794669567577356961152191e-03
            7.05875580731421203850572965166066e-01 2.88675824737542191522621237709245e-02 2.65256836794824590874952718877466e-01 2.29625688585843173994405219673354e-03
            6.25943112845752769324292330566095e-01 3.67127560052765977705035993494675e-02 3.37344131148970660660779685713351e-01 3.10138122358581767162855058472815e-03
            5.42387506520867757053849800286116e-01 4.49135315910860963284356728308921e-02 4.12698961888046167434396238604677e-01 3.90654127343776583908274702139352e-03
            4.57612493479132464990755124745192e-01 5.32339889226522491871840259136661e-02 4.89153517598215292760954753248370e-01 4.63024766721625673226991892761362e-03
            3.74056887154247230675707669433905e-01 6.14347645084617755006917150240042e-02 5.64508348337291021579176231171004e-01 5.18982081036291863218590236783712e-03
            2.94124419268578907171729497349588e-01 6.92799380399841402411453827880905e-02 6.36595642691436980342700735491235e-01 5.51083676372261799247809221924399e-03
            2.20114584463026341509817029873375e-01 7.65438198197804925415610455274873e-02 7.03341595717193124315258501155768e-01 5.53593212001406934780201041235159e-03
            1.54156478469823443600006385167944e-01 8.30174443807957102015038231002109e-02 7.62826077149380776809550752659561e-01 5.23195371793150083156165308650998e-03
            9.81475205137383732711953143734718e-02 8.85145847307436811624015149391198e-02 8.13337894755517987199766594130779e-01 4.59462105191824326383409626828325e-03
            5.36987667512220934185052101383917e-02 9.28771197024603478720194971174351e-02 8.53424113546317530953899677115260e-01 3.65015328480428752344622012060427e-03
            2.20880252143010324950012090994278e-02 9.59796356059098021429321079267538e-02 8.81932339179789193117642298602732e-01 2.45372205563344205833664624094581e-03
            4.21741578953443996269356830453034e-03 9.77335916110200764173043808114016e-02 8.98048992599445483620002050884068e-01 1.08638116943747986727075716117952e-03
            9.95782584210465393503852737921989e-01 2.26470026775140012348028206723427e-04 3.99094576275946678195483130480170e-03 3.48362681281367705081334143479843e-06
            9.77911974785698800971545097127091e-01 1.18609971397786958559328329698701e-03 2.09019255003233296597020540730227e-02 4.19614023509517979179835045489000e-05
            9.46301233248777906581494789861608e-01 2.88355755060215558291392135004116e-03 5.08152092006199374019104197941488e-02 1.56824355141451987410636248476692e-04
            9.01852479486261682239955916884355e-01 5.27040081127802279709060329082604e-03 9.28771197024602923608682658596081e-02 3.78582930901072292964593124509065e-04
            8.45843521530176500888842383574229e-01 8.27801278054083981838839889633164e-03 1.45878465689282671435833549367089e-01 7.21943397762896685657119100198997e-04
            7.79885415536973658490182970126625e-01 1.18198817296222228462321268693813e-02 2.08294702733404113459414475073572e-01 1.18297693963182663605404609796778e-03
            7.05875580731421203850572965166066e-01 1.57941185861420613223415898573876e-02 2.78330300682436748704873252791003e-01 1.73855315600468388406718567296139e-03
            6.25943112845752769324292330566095e-01 2.00863935349841274402571400514717e-02 3.53970493619263082418768817660748e-01 2.34813280144964249898653640968860e-03
            5.42387506520867757053849800286116e-01 2.45732265497810610510587281396511e-02 4.33039266929351185364538423527847e-01 2.95773948543164282715123292177850e-03
            4.57612493479132464990755124745192e-01 2.91255402014410184896586741842839e-02 5.13261966319426554683502672560280e-01 3.50567558207354711130587787693003e-03
            3.74056887154247230675707669433905e-01 3.36123732162379659782480700869201e-02 5.92330739629514768651574740943033e-01 3.92934231554069573338905385639919e-03
            2.94124419268578907171729497349588e-01 3.79046481650800182183758124665474e-02 6.67970932566341102365470305812778e-01 4.17239147187788757958903929079497e-03
            2.20114584463026341509817029873375e-01 4.18788850215998653681026553385891e-02 7.38006530515373793122080314788036e-01 4.19139178980845478889039057435184e-03
            1.54156478469823443600006385167944e-01 4.54207539706812518653933352652530e-02 8.00422767559495262901236856123433e-01 3.96124218696890066648830242002077e-03
            9.81475205137383732711953143734718e-02 4.84283659399440749582232967895834e-02 8.53424113546317641976202139630914e-01 3.47870178622101088039553218322908e-03
            5.36987667512220934185052101383917e-02 5.08152092006199374019104197941488e-02 8.95486024048157913668433138809633e-01 2.76362176735508264721619120507512e-03
            2.20880252143010324950012090994278e-02 5.25126670372442344580932171993481e-02 9.25399307748454802435844612773508e-01 1.85777394944427109643725337662090e-03
            4.21741578953443996269356830453034e-03 5.34722967244469563063979933303926e-02 9.42310287486018527403075495385565e-01 8.22526182667714686296001058707361e-04
            9.95782584210465393503852737921989e-01 9.31543862984316882631941059145220e-05 4.12426140323617507893461109347299e-03 2.26608152699167430822171573345791e-06
            9.77911974785698800971545097127091e-01 4.87880857867600630471682787714371e-04 2.16001443564335995906056098192494e-02 2.72956788495251147545154341012008e-05
            9.46301233248777906581494789861608e-01 1.18609971397786351406111737816218e-03 5.25126670372442275191993132921198e-02 1.02013445545102223463787560753246e-04
            9.01852479486261682239955916884355e-01 2.16788490782858110292319331335875e-03 9.59796356059097327539930688544700e-02 2.46266271402467370760930309714354e-04
            8.45843521530176500888842383574229e-01 3.40501218338931438064931533915569e-03 1.50751466286434182562103956115607e-01 4.69620509058702004408225594289661e-04
            7.79885415536973658490182970126625e-01 4.86189649165471877295674474339648e-03 2.15252687971371620134775071164768e-01 7.69520483622539576244736547749881e-04
            7.05875580731421203850572965166066e-01 6.49662758894601569809035268576736e-03 2.87627791679632760502016708414885e-01 1.13091998718807778927242235766926e-03
            6.25943112845752769324292330566095e-01 8.26217795504596841382749516924378e-03 3.65794709199201262261880174264661e-01 1.52744844674981036618255370029829e-03
            5.42387506520867757053849800286116e-01 1.01077562943462390815385631981371e-02 4.47504737184786005599335112492554e-01 1.92399449474239380569740021087455e-03
            4.57612493479132464990755124745192e-01 1.19802689199547882092922179708694e-02 5.30407237600912795372209984634537e-01 2.28042413927405504189827034622340e-03
            3.74056887154247230675707669433905e-01 1.38258472592550640811737139301840e-02 6.12117265586497705243118616635911e-01 2.55601719498813181527530069558907e-03
            2.94124419268578907171729497349588e-01 1.55913976253550124601021664716427e-02 6.90284183106066095980679619970033e-01 2.71411943524554107737278840772888e-03
            2.20114584463026341509817029873375e-01 1.72261287226463111199592503908207e-02 7.62659286814327375125799335364718e-01 2.72647904544002410778813505487506e-03
            1.54156478469823443600006385167944e-01 1.86830130309117159459475487892632e-02 8.27160508499264812698470450413879e-01 2.57676789913674269758003809727143e-03
            9.81475205137383732711953143734718e-02 1.99201403064724544278440987454815e-02 8.81932339179789193117642298602732e-01 2.26287782223761387706550962661822e-03
            5.36987667512220934185052101383917e-02 2.09019255003231700651422642067701e-02 9.25399307748454691413542150257854e-01 1.79772190625012047437980555741888e-03
            2.20880252143010324950012090994278e-02 2.16001443564334365265988679993825e-02 9.56311830429265596897892010019859e-01 1.20847250706563940844673332719594e-03
            4.21741578953443996269356830453034e-03 2.19948708280026036798737720801000e-02 9.73787713382462949418538755708141e-01 5.35049099161351337031189423498745e-04
            9.95782584210465393503852737921989e-01 1.77865959418151070917362560619424e-05 4.19962919359279096553905219479930e-03 9.85298067232842572036330755502753e-07
            9.77911974785698800971545097127091e-01 9.31543862984284763142581176076362e-05 2.19948708280027702133274658535811e-02 1.18682312590707993870840428196622e-05
            9.46301233248777906581494789861608e-01 2.26470026775131067680105201311846e-04 5.34722967244469632452918972376210e-02 4.43557080935161589935189496713974e-05
            9.01852479486261682239955916884355e-01 4.13928902718295552848326579464810e-04 9.77335916110200209061531495535746e-02 1.07077207217523567720288135696194e-04
            8.45843521530176500888842383574229e-01 6.50141966357659279764080739738574e-04 1.53506336503465845577665049859206e-01 2.04192203324087532710293113602518e-04
            7.79885415536973658490182970126625e-01 9.28314724021179126997593833436895e-04 2.19186269739005151757638145681995e-01 3.34589482407512853016917642179351e-04
            7.05875580731421203850572965166066e-01 1.24044496991095173563102171954142e-03 2.92883974298667859592626427911455e-01 4.91726914631654432076557359465596e-04
            6.25943112845752769324292330566095e-01 1.57755342206842446607650831680303e-03 3.72479333732178785609789883892518e-01 6.64138507134092738980191139575027e-04
            5.42387506520867757053849800286116e-01 1.92994215548711800164993768191835e-03 4.55682551323645101959414205339272e-01 8.36557747131431842825544098474211e-04
            4.57612493479132464990755124745192e-01 2.28747363404732066000102364000668e-03 5.40100032886820180522136070067063e-01 9.91534271885047106820376683344875e-04
            3.74056887154247230675707669433905e-01 2.63986236746601527977662549062643e-03 6.23303250478286718916365316545125e-01 1.11136284023243404710246107214289e-03
            2.94124419268578907171729497349588e-01 2.97697081962348779338167759078715e-03 7.02898609911797644933528772526188e-01 1.18010610030287363923784393904270e-03
            2.20114584463026341509817029873375e-01 3.28910106551326061885553997399256e-03 7.76596314471460380524092670384562e-01 1.18548009055491126648163024981386e-03
            1.54156478469823443600006385167944e-01 3.56727382317678035766883581914044e-03 8.42276247706999714459641381836263e-01 1.12038529968405361526839758568030e-03
            9.81475205137383732711953143734718e-02 3.80348688681614484352611071926731e-03 8.98048992599445483620002050884068e-01 9.83905088178664002582229919369183e-04
            5.36987667512220934185052101383917e-02 3.99094576275930892211851741535611e-03 9.42310287486018638425377957901219e-01 7.81654101387012074757254520562810e-04
            2.20880252143010324950012090994278e-02 4.12426140323601201492786927360612e-03 9.73787713382463060440841218223795e-01 5.25446949429272231434084883261448e-04
            4.21741578953443996269356830453034e-03 4.19962919359262529944709640972178e-03 9.91582955016872880094069842016324e-01 2.32640722321324466262057462095925e-04
            ];
        
    case  35
        
        % ALG. DEG.:   35
        % PTS CARD.:  361
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.8e-15
        
        xyw_bar=[
            9.96203421921792342530466157768387e-01 3.78216407310378579728626924350010e-03 1.44140051038716722475729881125517e-05 7.18998236483455251024776017088058e-07
            9.80104076067414897721619126969017e-01 1.98203875039369079025242115221772e-02 7.55364286481943758566615088057006e-05 8.67626415872991205971345024527963e-06
            9.51577951807409005979820904030930e-01 4.82382101059210879889782574991841e-02 1.83838086669906031200838469885639e-04 3.25330859259954940400361156349618e-05
            9.11357328268571409424225748807658e-01 8.83061329071393452672111834544921e-02 3.36538824289245308563067737850361e-04 7.89167723463550990553247443948237e-05
            8.60483088667614692823804034560453e-01 1.38987224485281529728908367360418e-01 5.29686847103777447287598079128657e-04 1.51465588105983848176902539783839e-04
            8.00272652330840550405355315888301e-01 1.98969067199380095223304465434921e-01 7.58280469779354371340218676778022e-04 2.50236631568340845907516545310045e-04
            7.32285370687980496917646178189898e-01 2.66698229819158028064407517376821e-01 1.01639949286147501794630443328060e-03 3.71505106415705621709932859886294e-04
            6.58282049981815053918410285405116e-01 3.40420591140215833547699730843306e-01 1.29735887796911253388998375157826e-03 5.07980880002682918215872742706551e-04
            5.80179322820112863823283078090753e-01 4.18226795200127676732648751567467e-01 1.59388197975945944406817034177948e-03 6.49424376622865606560608409125734e-04
            5.00000000000000111022302462515654e-01 4.98101710960896060242930616368540e-01 1.89828903910382873476692111580633e-03 7.83601899103001557980729963048816e-04
            4.19820677179887136176716921909247e-01 5.77976626721664610286666174943093e-01 2.20269609844825353661690314766020e-03 8.97484606005920400317865670558604e-04
            3.41717950018184946081589714594884e-01 6.55782830781576508982766426925082e-01 2.49921920023854493564385848003440e-03 9.78569299686884563199695818980217e-04
            2.67714629312019669615807515583583e-01 7.29505192102634203443756177875912e-01 2.78017858534618245158753779833205e-03 1.01618561250544714794075140673613e-03
            1.99727347669159671639249609143008e-01 7.97234354722411997506981151673244e-01 3.03829760842833085376923918374814e-03 1.00265454477095833710209582534389e-03
            1.39516911332385140642742271666066e-01 8.57216197436511007090587099810364e-01 3.26689123110390777782185978139751e-03 9.34177626465556307580606620888375e-04
            8.86426717314284795534717886766884e-02 9.07897289014653163796708668087376e-01 3.46003925391835664981954323593527e-03 8.11362940628210997477676169609140e-04
            4.84220481925909940201790959690697e-02 9.47965211815871233724806188547518e-01 3.61273999153777225501471548341215e-03 6.39332048663113303860194847061393e-04
            1.98959239325848247226247167418478e-02 9.76383034417855566466926120483549e-01 3.72104164955955329929793151677586e-03 4.27406231337741464304524896533621e-04
            3.79657807820771298068507348943967e-03 9.92421257848688620484267630672548e-01 3.78216407310372204619852709583938e-03 1.88661602312875778314146812952856e-04
            9.96203421921792342530466157768387e-01 3.72104164955957238125616726165390e-03 7.55364286480850882776749699587526e-05 1.65562124269206578503735478868908e-06
            9.80104076067414897721619126969017e-01 1.95000761434538337490973702870178e-02 3.95847789131268529283502743965073e-04 1.99786404465424458261075396636741e-05
            9.51577951807409005979820904030930e-01 4.74586468050912368976668176401290e-02 9.63401387499757122512278328940738e-04 7.49132131572951924774333987144814e-05
            9.11357328268571409424225748807658e-01 8.68790438774790368947620322614966e-02 1.76362785394955368101221893084585e-03 1.81719895921227071397541230624029e-04
            8.60483088667614692823804034560453e-01 1.36741093477206948625379823170078e-01 2.77581785517835855081614226946840e-03 3.48776439886141532715352342464143e-04
            8.00272652330840550405355315888301e-01 1.95753587552676827643338697271247e-01 3.97376011648262195130598684045253e-03 5.76214324183236701812371727271511e-04
            7.32285370687980496917646178189898e-01 2.62388199411587363929498906145454e-01 5.32642990043213915285491566464771e-03 8.55456543201927655829808205822928e-04
            6.58282049981815053918410285405116e-01 3.34919155678224222949523891657009e-01 6.79879433996072313206582293787505e-03 1.16971627069241608000427223856832e-03
            5.80179322820112863823283078090753e-01 4.11467956921389710345238199806772e-01 8.35272025849742583147872210247442e-03 1.49541506348828184477794955853369e-03
            5.00000000000000111022302462515654e-01 4.90052038033707337838507100968855e-01 9.94796196629255113919043651549146e-03 1.80438265928713073918410181306626e-03
            4.19820677179887136176716921909247e-01 5.68636119146025187376380927162245e-01 1.15432036740876764469021509285085e-02 2.06661783478061808830283929694360e-03
            3.41717950018184946081589714594884e-01 6.45184920389190730283246466569835e-01 1.30971295926243236351638188352808e-02 2.25332975492657577518129663474156e-03
            2.67714629312019669615807515583583e-01 7.17715876655827478280968989565736e-01 1.45694940321529076143747261085082e-02 2.33994800155644128852872043466959e-03
            1.99727347669159671639249609143008e-01 7.84350488514737764766948657779722e-01 1.59221638161025635938017330772709e-02 2.30879031292692202259608791337087e-03
            1.39516911332385140642742271666066e-01 8.43362982590208143385268613201333e-01 1.71201060774067714831403463904280e-02 2.15111003663722962289672757663084e-03
            8.86426717314284795534717886766884e-02 8.93225032189935985726947365037631e-01 1.81322960786355347195808462856803e-02 1.86830739197241160048923447334346e-03
            4.84220481925909940201790959690697e-02 9.32645429262323633068376693699975e-01 1.89325225450853729114442103309557e-02 1.47217568443207833320252042597076e-03
            1.98959239325848247226247167418478e-02 9.60603999923961171525377267244039e-01 1.95000761434539482408467847562861e-02 9.84178819857240523225061501477740e-04
            3.79657807820771298068507348943967e-03 9.76383034417855344422321195452241e-01 1.98203875039369981081449623161461e-02 4.34426874253824733991546924727345e-04
            9.96203421921792342530466157768387e-01 3.61273999153780478107989004854517e-03 1.83838086669852688453952183067486e-04 2.55078844149308762621029822437180e-06
            9.80104076067414897721619126969017e-01 1.89325225450852896447173634442152e-02 9.63401387499812633663509586767759e-04 3.07807630233850623800023749154775e-05
            9.51577951807409005979820904030930e-01 4.60773534414253907454117609177047e-02 2.34469475116560327476733505136508e-03 1.15417556449097516370512483963040e-04
            9.11357328268571409424225748807658e-01 8.43504120089293846218581052198715e-02 4.29225972249920595391614597247099e-03 2.79972857410000063732580244391102e-04
            8.60483088667614692823804034560453e-01 1.32761216728167097400614693469834e-01 6.75569460421820977558127196971327e-03 5.37354129426417478389288451268158e-04
            8.00272652330840550405355315888301e-01 1.90056140414944979566413962857041e-01 9.67120725421447002823072125465842e-03 8.87763940235166380821052456440157e-04
            7.32285370687980496917646178189898e-01 2.54751338629611245867323532365845e-01 1.29632906824082572150302894442575e-02 1.31798783823949936140629279890391e-03
            6.58282049981815053918410285405116e-01 3.25171266974131012439386267942609e-01 1.65466830440539336422034466522746e-02 1.80216263609729609486420631725423e-03
            5.80179322820112863823283078090753e-01 3.99492100117236459411174109845888e-01 2.03285770626506767655428120633587e-02 2.30396141389086405934505386028377e-03
            5.00000000000000111022302462515654e-01 4.75788975903704391967607989499811e-01 2.42110240962954970100895479845349e-02 2.77998271141790664201520755227648e-03
            4.19820677179887136176716921909247e-01 5.52085851690172546568646794185042e-01 2.80934711299403172546362839057110e-02 3.18400413694277292092627007491501e-03
            3.41717950018184946081589714594884e-01 6.26406684833277993540434636088321e-01 3.18753651485370603779756493167952e-02 3.47166812404098128197471595512980e-03
            2.67714629312019669615807515583583e-01 6.96826613177797593579043677891605e-01 3.54587575101827923163000377826393e-02 3.60511943321034037529537741306740e-03
            1.99727347669159671639249609143008e-01 7.61521811392463776613226400513668e-01 3.87508409383765517475239903433248e-02 3.55711529435875873847439088137889e-03
            1.39516911332385140642742271666066e-01 8.18816735079242130623811135592405e-01 4.16663535883727842445978239993565e-02 3.31417988386765787964249874164580e-03
            8.86426717314284795534717886766884e-02 8.67227539798479774013628684770083e-01 4.41297884700917464328995265532285e-02 2.87847049657945393275060297355594e-03
            4.84220481925909940201790959690697e-02 9.05500598365983622173303047020454e-01 4.60773534414253838065178570104763e-02 2.26815688447588002796262429683338e-03
            1.98959239325848247226247167418478e-02 9.32645429262323855112981618731283e-01 4.74586468050912646532424332690425e-02 1.51630813456594439048163991401452e-03
            3.79657807820771298068507348943967e-03 9.47965211815871233724806188547518e-01 4.82382101059211088056599692208692e-02 6.69314346147666474652615509199904e-04
            9.96203421921792342530466157768387e-01 3.46003925391840869152382254014810e-03 3.36538824289248778010019691464549e-04 3.38001644175229746161330003639822e-06
            9.80104076067414897721619126969017e-01 1.81322960786354375750661915844830e-02 1.76362785394966470331468144649989e-03 4.07871869796555281545424875844219e-05
            9.51577951807409005979820904030930e-01 4.41297884700917325551117187387717e-02 4.29225972249926146506737723029801e-03 1.52938296300446314795670743613698e-04
            9.11357328268571409424225748807658e-01 8.07851484797428298634258680976927e-02 7.85752325168576071234838309464976e-03 3.70988375945538527249378812911118e-04
            8.60483088667614692823804034560453e-01 1.27149759560165859229030616006639e-01 1.23671517722194479471653494329075e-02 7.12040937209846994462858482677348e-04
            8.00272652330840550405355315888301e-01 1.82022981953933182808214041870087e-01 1.77043657152262667864306422416121e-02 1.17636439995519640443200071189267e-03
            7.32285370687980496917646178189898e-01 2.43983689308213069768527248015744e-01 2.37309400038064333138265737943584e-02 1.74644846699692360761368981059150e-03
            6.58282049981815053918410285405116e-01 3.11427157949986266327613293469767e-01 3.02907920681986797539764211251168e-02 2.38802216665016595054105152939883e-03
            5.80179322820112863823283078090753e-01 3.82606650706564355157723866796005e-01 3.72140264733227810189930551132420e-02 3.05294917188650514333625451968146e-03
            5.00000000000000111022302462515654e-01 4.55678664134285593689810411888175e-01 4.43213358657142952878871255961712e-02 3.68371877476422228403496461623945e-03
            4.19820677179887136176716921909247e-01 5.28750677562007109777653113269480e-01 5.14286452581057540456299648212735e-02 4.21908228781781892424707081090673e-03
            3.41717950018184946081589714594884e-01 5.99930170318585198607763686595717e-01 5.83518796632298553106465988093987e-02 4.60026207924059578335818088135056e-03
            2.67714629312019669615807515583583e-01 6.67373638960358284144547269534087e-01 6.49117317276221017507964461401571e-02 4.77709666568782581608232362668787e-03
            1.99727347669159671639249609143008e-01 7.29334346314638004571406781906262e-01 7.09383060162023237893436089507304e-02 4.71348700839475248658416717262298e-03
            1.39516911332385140642742271666066e-01 7.84207568708405799995375673461240e-01 7.62755199592091148730332861305214e-02 4.39157646952500514692108168901541e-03
            8.86426717314284795534717886766884e-02 8.30572179788828690583102343225619e-01 8.07851484797428298634258680976927e-02 3.81422365229257698954201671881492e-03
            4.84220481925909940201790959690697e-02 8.67227539798479662991326222254429e-01 8.43504120089293429884946817765012e-02 3.00550505768900971456658588465416e-03
            1.98959239325848247226247167418478e-02 8.93225032189936096749249827553285e-01 8.68790438774790230169742244470399e-02 2.00924010091392589785752598174895e-03
            3.79657807820771298068507348943967e-03 9.07897289014653052774406205571722e-01 8.83061329071392897560599521966651e-02 8.86899696532881523167168857924025e-04
            9.96203421921792342530466157768387e-01 3.26689123110392946186530949148619e-03 5.29686847103728007668532740126466e-04 4.12172922919727556707609333885856e-06
            9.80104076067414897721619126969017e-01 1.71201060774066951553074034109159e-02 2.77581785517840712307346962006704e-03 4.97375511770072148919584476001177e-05
            9.51577951807409005979820904030930e-01 4.16663535883727911834917279065849e-02 6.75569460421820283668736806248489e-03 1.86499165607129752889742357879754e-04
            9.11357328268571409424225748807658e-01 7.62755199592091842619723252028052e-02 1.23671517722194063138019259895373e-02 4.52398282428004120964831757945035e-04
            8.60483088667614692823804034560453e-01 1.20051942784656640039742114822729e-01 1.94649685477286671364538506168174e-02 8.68291617469571551726770231027785e-04
            8.00272652330840550405355315888301e-01 1.71862005013748792014993682641943e-01 2.78653426554106575796510014697560e-02 1.43450649280533223776934459436916e-03
            7.32285370687980496917646178189898e-01 2.30363911111912067486429123164271e-01 3.73507182001074355959246986458311e-02 2.12969014138172148958849660971282e-03
            6.58282049981815053918410285405116e-01 2.94042517084813381789132336052717e-01 4.76754329333715642924573785421671e-02 2.91205114941695854258174591677744e-03
            5.80179322820112863823283078090753e-01 3.61248592986278871386218725092476e-01 5.85720841936082647904981968167704e-02 3.72289012608903588028019804312407e-03
            5.00000000000000111022302462515654e-01 4.30241544333807235389599554764573e-01 6.97584556661926535880979827197734e-02 4.49207618002503503845668220151310e-03
            4.19820677179887136176716921909247e-01 4.99234495681335821437585309467977e-01 8.09448271387770423856977686227765e-02 5.14492017591245461710647290942688e-03
            3.41717950018184946081589714594884e-01 5.66440571582801366545822929765563e-01 9.18414783990136873725873556395527e-02 5.60974628399850405724436086529749e-03
            2.67714629312019669615807515583583e-01 6.30119177555702569826223680138355e-01 1.02166193132277816069120035535889e-01 5.82538555565680852416088342238254e-03
            1.99727347669159671639249609143008e-01 6.88621083653865651008629811258288e-01 1.11651568676974677352120579598704e-01 5.74781735791509654914799298808248e-03
            1.39516911332385140642742271666066e-01 7.40431145882958219317515613511205e-01 1.20051942784656695550893346080557e-01 5.35526658187265251070963145707537e-03
            8.86426717314284795534717886766884e-02 7.84207568708405688973073210945586e-01 1.27149759560165831473455000377726e-01 4.65121912430685718192746591626019e-03
            4.84220481925909940201790959690697e-02 8.18816735079241908579206210561097e-01 1.32761216728167097400614693469834e-01 3.66503484768695465823018508899622e-03
            1.98959239325848247226247167418478e-02 8.43362982590208143385268613201333e-01 1.36741093477206976380955438798992e-01 2.45014892534629836681747683257981e-03
            3.79657807820771298068507348943967e-03 8.57216197436510785045982174779056e-01 1.38987224485281557484483982989332e-01 1.08152148534242749031153429939422e-03
            9.96203421921792342530466157768387e-01 3.03829760842841195209174109947980e-03 7.58280469779245517442101132132848e-04 4.75669917446964873869582821064306e-06
            9.80104076067414897721619126969017e-01 1.59221638161024837965218381441446e-02 3.97376011648261848185903488683834e-03 5.73998328050987378099248581442282e-05
            9.51577951807409005979820904030930e-01 3.87508409383765795030996059722384e-02 9.67120725421441451707948999683140e-03 2.15230156507753626849918648922255e-04
            9.11357328268571409424225748807658e-01 7.09383060162024209338582636519277e-02 1.77043657152261696419159875404148e-02 5.22092164937256277869315201911604e-04
            8.60483088667614692823804034560453e-01 1.11651568676974691229908387413161e-01 2.78653426554106159462875780263857e-02 1.00205563984143652088432130398132e-03
            8.00272652330840550405355315888301e-01 1.59836334262202117706408444064436e-01 3.98910134069573318882362400472630e-02 1.65549832865352565709760046530619e-03
            7.32285370687980496917646178189898e-01 2.14244696467297629505210920797253e-01 5.34699328447218735771429010128486e-02 2.45777797959801985391847800599407e-03
            6.58282049981815053918410285405116e-01 2.73467530210110443267268465206143e-01 6.82504198080745028143212493887404e-02 3.36066503357929828935124483280106e-03
            5.80179322820112863823283078090753e-01 3.35971006830077856264438196376432e-01 8.38496703498092799122787255328149e-02 4.29641720857480008399775783800578e-03
            5.00000000000000111022302462515654e-01 4.00136326165420164180375195428496e-01 9.98636738345797247973223420558497e-02 5.18409965065591126637256635945050e-03
            4.19820677179887136176716921909247e-01 4.64301645500762694140917119511869e-01 1.15877677319350169682365958578885e-01 5.93751704505858513005733811951359e-03
            3.41717950018184946081589714594884e-01 5.26805122120730051626935619424330e-01 1.31476927861085002291474665980786e-01 6.47395159513587496735720350216070e-03
            2.67714629312019669615807515583583e-01 5.86027955863542837633417548204307e-01 1.46257414824437548261926167469937e-01 6.72281101516140119217501691650796e-03
            1.99727347669159671639249609143008e-01 6.40436318068638210654341946792556e-01 1.59836334262202117706408444064436e-01 6.63329310613685382141957447288405e-03
            1.39516911332385140642742271666066e-01 6.88621083653865984075537198805250e-01 1.71862005013748930792871760786511e-01 6.18026821087895496947828277711778e-03
            8.86426717314284795534717886766884e-02 7.29334346314638226616011706937570e-01 1.82022981953933293830516504385741e-01 5.36775924341267255540666170077202e-03
            4.84220481925909940201790959690697e-02 7.61521811392463998657831325544976e-01 1.90056140414945007321989578485955e-01 4.22964907808615932227613853910952e-03
            1.98959239325848247226247167418478e-02 7.84350488514738208856158507842338e-01 1.95753587552676910910065544157987e-01 2.82760480430496946926965939894671e-03
            3.79657807820771298068507348943967e-03 7.97234354722412219551586076704552e-01 1.98969067199380122978880081063835e-01 1.24813447716492055897252733132063e-03
            9.96203421921792342530466157768387e-01 2.78017858534619589569447661858703e-03 1.01639949286146157383936561302562e-03 5.26847616278834051777848687669881e-06
            9.80104076067414897721619126969017e-01 1.45694940321529041449277741548940e-02 5.32642990043219813345309887608892e-03 6.35755257563489390186181604569526e-05
            9.51577951807409005979820904030930e-01 3.54587575101827506829366143392690e-02 1.29632906824082433372424816298007e-02 2.38386937555438874290075768058728e-04
            9.11357328268571409424225748807658e-01 6.49117317276221988953111008413543e-02 2.37309400038063916804631503509881e-02 5.78264469721732313374495948465892e-04
            8.60483088667614692823804034560453e-01 1.02166193132277885458059074608173e-01 3.73507182001074217181368908313743e-02 1.10986759066613606926343660319390e-03
            8.00272652330840550405355315888301e-01 1.46257414824437548261926167469937e-01 5.34699328447219013327185166417621e-02 1.83361468996397040942536449392719e-03
            7.32285370687980496917646178189898e-01 1.96043506564347491671895795661840e-01 7.16711227476720114104580261482624e-02 2.72221223667817608521390937426077e-03
            6.58282049981815053918410285405116e-01 2.50235055699803365936162435900769e-01 9.14828943183815801454272786941146e-02 3.72224161568991945142625965559091e-03
            5.80179322820112863823283078090753e-01 3.07428540211152623307100384408841e-01 1.12392136968734512869616537500406e-01 4.75867209981672371205130289695262e-03
            5.00000000000000111022302462515654e-01 3.66142685343990192947671857837122e-01 1.33857314656009696030025679647224e-01 5.74186099082989349906558373959342e-03
            4.19820677179887136176716921909247e-01 4.24856830476827873610545793781057e-01 1.55322492343284990212737284309696e-01 6.57633915256547463279446219530655e-03
            3.41717950018184946081589714594884e-01 4.82050314988177130981483742289129e-01 1.76231734993637922936926543115987e-01 7.17048911587346484569627236282940e-03
            2.67714629312019669615807515583583e-01 5.36241864123632949734599151270231e-01 1.96043506564347436160744564404013e-01 7.44612351573773617596785356909095e-03
            1.99727347669159671639249609143008e-01 5.86027955863542726611115085688652e-01 2.14244696467297601749635305168340e-01 7.34697430479545107034899942277661e-03
            1.39516911332385140642742271666066e-01 6.30119177555702791870828605169663e-01 2.30363911111912122997580354422098e-01 6.84520810637235296164382347683386e-03
            8.86426717314284795534717886766884e-02 6.67373638960358395166849732049741e-01 2.43983689308213125279678479273571e-01 5.94528066296299510101741603307346e-03
            4.84220481925909940201790959690697e-02 6.96826613177797704601346140407259e-01 2.54751338629611301378474763623672e-01 4.68472033389438768935519874503370e-03
            1.98959239325848247226247167418478e-02 7.17715876655827700325573914597044e-01 2.62388199411587419440650137403281e-01 3.13182901900191986571675428763228e-03
            3.79657807820771298068507348943967e-03 7.29505192102634314466058640391566e-01 2.66698229819158028064407517376821e-01 1.38242224275846557757230215202071e-03
            9.96203421921792342530466157768387e-01 2.49921920023859307422031683643127e-03 1.29735887796906439531352539518139e-03 5.64380382436514189563097829149818e-06
            9.80104076067414897721619126969017e-01 1.30971295926243392476751026265447e-02 6.79879433996076303070577040443823e-03 6.81046633434529585272854124156083e-05
            9.51577951807409005979820904030930e-01 3.18753651485370395612939375951100e-02 1.65466830440539544588851583739597e-02 2.55369687226984169031718252540486e-04
            9.11357328268571409424225748807658e-01 5.83518796632299385773734456961392e-02 3.02907920681986519984008054962032e-02 6.19460186374408481972675399873651e-04
            8.60483088667614692823804034560453e-01 9.18414783990137290059507790829230e-02 4.76754329333715781702451863566239e-02 1.18893485691037251866797319621583e-03
            8.00272652330840550405355315888301e-01 1.31476927861084946780323434722959e-01 6.82504198080745028143212493887404e-02 1.96424189459628965870319206032946e-03
            7.32285370687980496917646178189898e-01 1.76231734993637922936926543115987e-01 9.14828943183815801454272786941146e-02 2.91614336999613133621922500537949e-03
            6.58282049981815053918410285405116e-01 2.24946792653554189511666550060909e-01 1.16771157364630756569923164533975e-01 3.98741511145484298084262775319075e-03
            5.80179322820112863823283078090753e-01 2.76360415998729902486985565701616e-01 1.43460261181157233689731356207631e-01 5.09768118256631770635056710716526e-03
            5.00000000000000111022302462515654e-01 3.29141024990907471448053911444731e-01 1.70858975009092417529643626039615e-01 6.15091271512329187953760367690847e-03
            4.19820677179887136176716921909247e-01 3.81921633983085151431424719703500e-01 1.98257688837027712391858358387253e-01 7.04483932597463509417412552693349e-03
            3.41717950018184946081589714594884e-01 4.33335257328260836651168119715294e-01 2.24946792653554217267242165689822e-01 7.68131669277918177618502681980317e-03
            2.67714629312019669615807515583583e-01 4.82050314988177075470332511031302e-01 2.50235055699803310425011204642942e-01 7.97658736156726076715894890867276e-03
            1.99727347669159671639249609143008e-01 5.26805122120729940604633156908676e-01 2.73467530210110387756117233948316e-01 7.87037473411889010321829118765891e-03
            1.39516911332385140642742271666066e-01 5.66440571582801477568125392281218e-01 2.94042517084813437300283567310544e-01 7.33286257650505191724166209610303e-03
            8.86426717314284795534717886766884e-02 5.99930170318585198607763686595717e-01 3.11427157949986321838764524727594e-01 6.36882405951633492890984200585081e-03
            4.84220481925909940201790959690697e-02 6.26406684833277993540434636088321e-01 3.25171266974131012439386267942609e-01 5.01846107291128214894504466769831e-03
            1.98959239325848247226247167418478e-02 6.45184920389190841305548929085489e-01 3.34919155678224278460675122914836e-01 3.35494136227543421693852998544116e-03
            3.79657807820771298068507348943967e-03 6.55782830781576508982766426925082e-01 3.40420591140215833547699730843306e-01 1.48090637586531146362156530216225e-03
            9.96203421921792342530466157768387e-01 2.20269609844823662306301237379103e-03 1.59388197975942084647082985782163e-03 5.87296073243102944097526751932037e-06
            9.80104076067414897721619126969017e-01 1.15432036740876695080082470212801e-02 8.35272025849743277037262600970280e-03 7.08699355893233014268631220211603e-05
            9.51577951807409005979820904030930e-01 2.80934711299403137851893319520968e-02 2.03285770626506802349897640169729e-02 2.65738532381744872214179853386895e-04
            9.11357328268571409424225748807658e-01 5.14286452581058303734629078007856e-02 3.72140264733227602023113433915569e-02 6.44612297503190621872359855615287e-04
            8.60483088667614692823804034560453e-01 8.09448271387770423856977686227765e-02 5.85720841936082647904981968167704e-02 1.23720950361675178195974833528226e-03
            8.00272652330840550405355315888301e-01 1.15877677319350141926790342949971e-01 8.38496703498093076678543411617284e-02 2.04399654469874939369256061638680e-03
            7.32285370687980496917646178189898e-01 1.55322492343285017968312899938610e-01 1.12392136968734485114040921871492e-01 3.03454833567906243074752126176463e-03
            6.58282049981815053918410285405116e-01 1.98257688837027712391858358387253e-01 1.43460261181157233689731356207631e-01 4.14931721623237274171325239535690e-03
            5.80179322820112863823283078090753e-01 2.43571276192108138181069421079883e-01 1.76249400987778997995647500829364e-01 5.30466372385511933384982796724216e-03
            5.00000000000000111022302462515654e-01 2.90089661410056376400490307787550e-01 2.09910338589943512577207229696796e-01 6.40065990397767434899600758058114e-03
            4.19820677179887136176716921909247e-01 3.36608046628004753397789272639784e-01 2.43571276192108110425493805450969e-01 7.33088286115065029119319817141331e-03
            3.41717950018184946081589714594884e-01 3.81921633983085151431424719703500e-01 2.76360415998729902486985565701616e-01 7.99320329230856357893841845907446e-03
            2.67714629312019669615807515583583e-01 4.24856830476827762588243331265403e-01 3.07428540211152623307100384408841e-01 8.30046291670312637311912595805552e-03
            1.99727347669159671639249609143008e-01 4.64301645500762583118614656996215e-01 3.35971006830077745242135733860778e-01 8.18993770893462931514950042810597e-03
            1.39516911332385140642742271666066e-01 4.99234495681335932459887771983631e-01 3.61248592986278982408521187608130e-01 7.63060079838468515583960183334966e-03
            8.86426717314284795534717886766884e-02 5.28750677562007109777653113269480e-01 3.82606650706564410668875098053832e-01 6.62741916220110395541187386925230e-03
            4.84220481925909940201790959690697e-02 5.52085851690172546568646794185042e-01 3.99492100117236459411174109845888e-01 5.22222701845187226715738759708074e-03
            1.98959239325848247226247167418478e-02 5.68636119146025298398683389677899e-01 4.11467956921389821367540662322426e-01 3.49116296267946209924692979598149e-03
            3.79657807820771298068507348943967e-03 5.77976626721664610286666174943093e-01 4.18226795200127732243799982825294e-01 1.54103602189646616237028720064473e-03
            9.96203421921792342530466157768387e-01 1.89828903910385692402340573892161e-03 1.89828903910380054551043649269104e-03 5.95001158435277681700093302397114e-06
            9.80104076067414897721619126969017e-01 9.94796196629252511833829686338504e-03 9.94796196629257716004257616759787e-03 7.17997202689045591365510357029223e-05
            9.51577951807409005979820904030930e-01 2.42110240962955039489834518917633e-02 2.42110240962954900711956440773065e-02 2.69224913653695380102298662094995e-04
            9.11357328268571409424225748807658e-01 4.43213358657143299823566451323131e-02 4.43213358657142605934176060600294e-02 6.53069348204651138958820588698018e-04
            8.60483088667614692823804034560453e-01 6.97584556661926674658857905342302e-02 6.97584556661926397103101749053167e-02 1.25344118821374612450514351991160e-03
            8.00272652330840550405355315888301e-01 9.98636738345797247973223420558497e-02 9.98636738345797247973223420558497e-02 2.07081294655624224138046329812823e-03
            7.32285370687980496917646178189898e-01 1.33857314656009779296752526533965e-01 1.33857314656009723785601295276138e-01 3.07436037344234035922907466442666e-03
            6.58282049981815053918410285405116e-01 1.70858975009092500796370472926355e-01 1.70858975009092445285219241668528e-01 4.20375456750543899375749390401324e-03
            5.80179322820112863823283078090753e-01 2.09910338589943623599509692212450e-01 2.09910338589943512577207229696796e-01 5.37425875057211815954971712017141e-03
            5.00000000000000111022302462515654e-01 2.50000000000000000000000000000000e-01 2.49999999999999888977697537484346e-01 6.48463395402358762364958266743997e-03
            4.19820677179887136176716921909247e-01 2.90089661410056487422792770303204e-01 2.90089661410056376400490307787550e-01 7.42706105738326523291181757713275e-03
            3.41717950018184946081589714594884e-01 3.29141024990907582470356373960385e-01 3.29141024990907471448053911444731e-01 8.09807086274116037782899724106755e-03
            2.67714629312019669615807515583583e-01 3.66142685343990248458823089094949e-01 3.66142685343990137436520626579295e-01 8.40936160821746832294554252484886e-03
            1.99727347669159671639249609143008e-01 4.00136326165420275202677657944150e-01 4.00136326165420053158072732912842e-01 8.29738635475560024068464315405436e-03
            1.39516911332385140642742271666066e-01 4.30241544333807568456506942311535e-01 4.30241544333807346411902017280227e-01 7.73071117183629467894956377449489e-03
            8.86426717314284795534717886766884e-02 4.55678664134285871245566568177310e-01 4.55678664134285649200961643146002e-01 6.71436820132378323761068728003920e-03
            4.84220481925909940201790959690697e-02 4.75788975903704614012212914531119e-01 4.75788975903704391967607989499811e-01 5.29074050918211351290931787616501e-03
            1.98959239325848247226247167418478e-02 4.90052038033707670905414488515817e-01 4.90052038033707448860809563484509e-01 3.53696559830525081649654772775193e-03
            3.79657807820771298068507348943967e-03 4.98101710960896282287535541399848e-01 4.98101710960896060242930616368540e-01 1.56125378662176736606348459446281e-03
            9.96203421921792342530466157768387e-01 1.59388197975947657446249561274954e-03 2.20269609844818067823091212176223e-03 5.87296073243105485196368514833054e-06
            9.80104076067414897721619126969017e-01 8.35272025849737725922139475187578e-03 1.15432036740877250191594782791071e-02 7.08699355893236131349877116036851e-05
            9.51577951807409005979820904030930e-01 2.03285770626506802349897640169729e-02 2.80934711299403137851893319520968e-02 2.65738532381746010626460963166551e-04
            9.11357328268571409424225748807658e-01 3.72140264733228157134625746493839e-02 5.14286452581057748623116765429586e-02 6.44612297503193440798008317926815e-04
            8.60483088667614692823804034560453e-01 5.85720841936082717293921007239987e-02 8.09448271387770423856977686227765e-02 1.23720950361675720297061076280443e-03
            8.00272652330840550405355315888301e-01 8.38496703498092521567031099039013e-02 1.15877677319350197437941574207798e-01 2.04399654469875850099080949462405e-03
            7.32285370687980496917646178189898e-01 1.12392136968734485114040921871492e-01 1.55322492343285017968312899938610e-01 3.03454833567907587485446008201961e-03
            6.58282049981815053918410285405116e-01 1.43460261181157233689731356207631e-01 1.98257688837027712391858358387253e-01 4.14931721623239008894801216342785e-03
            5.80179322820112863823283078090753e-01 1.76249400987778997995647500829364e-01 2.43571276192108138181069421079883e-01 5.30466372385514275261675365413794e-03
            5.00000000000000111022302462515654e-01 2.09910338589943512577207229696796e-01 2.90089661410056376400490307787550e-01 6.40065990397770210457162320949465e-03
            4.19820677179887136176716921909247e-01 2.43571276192108138181069421079883e-01 3.36608046628004697886638041381957e-01 7.33088286115068151621576575394101e-03
            3.41717950018184946081589714594884e-01 2.76360415998729902486985565701616e-01 3.81921633983085151431424719703500e-01 7.99320329230859827340793799521634e-03
            2.67714629312019669615807515583583e-01 3.07428540211152623307100384408841e-01 4.24856830476827762588243331265403e-01 8.30046291670316280231212147100450e-03
            1.99727347669159671639249609143008e-01 3.35971006830077745242135733860778e-01 4.64301645500762583118614656996215e-01 8.18993770893466400961901996424785e-03
            1.39516911332385140642742271666066e-01 3.61248592986278982408521187608130e-01 4.99234495681335932459887771983631e-01 7.63060079838471898294738338108800e-03
            8.86426717314284795534717886766884e-02 3.82606650706564410668875098053832e-01 5.28750677562007109777653113269480e-01 6.62741916220113257834922748656936e-03
            4.84220481925909940201790959690697e-02 3.99492100117236459411174109845888e-01 5.52085851690172546568646794185042e-01 5.22222701845189568592431328397652e-03
            1.98959239325848247226247167418478e-02 4.11467956921389821367540662322426e-01 5.68636119146025298398683389677899e-01 3.49116296267947727807734459304356e-03
            3.79657807820771298068507348943967e-03 4.18226795200127676732648751567467e-01 5.77976626721664610286666174943093e-01 1.54103602189647288442375661077222e-03
            9.96203421921792342530466157768387e-01 1.29735887796911990646475665300841e-03 2.49921920023853756306908557860424e-03 5.64380382436515121299339808880191e-06
            9.80104076067414897721619126969017e-01 6.79879433996070665219280115820766e-03 1.30971295926243964935498098611788e-02 6.81046633434530669475026609660517e-05
            9.51577951807409005979820904030930e-01 1.65466830440539509894382064203455e-02 3.18753651485370465001878415023384e-02 2.55369687226984602712587246742260e-04
            9.11357328268571409424225748807658e-01 3.02907920681987109789989887076445e-02 5.83518796632298830662222144383122e-02 6.19460186374409457754630636827642e-04
            8.60483088667614692823804034560453e-01 4.76754329333715851091390902638523e-02 9.18414783990137151281629712684662e-02 1.18893485691037447023188367012381e-03
            8.00272652330840550405355315888301e-01 6.82504198080744611809578259453701e-02 1.31476927861085002291474665980786e-01 1.96424189459629269446927501974187e-03
            7.32285370687980496917646178189898e-01 9.14828943183815940232150865085714e-02 1.76231734993637922936926543115987e-01 2.91614336999613610670878394159899e-03
            6.58282049981815053918410285405116e-01 1.16771157364630742692135356719518e-01 2.24946792653554217267242165689822e-01 3.98741511145484905237479367201558e-03
            5.80179322820112863823283078090753e-01 1.43460261181157233689731356207631e-01 2.76360415998729902486985565701616e-01 5.09768118256632637996794699120073e-03
            5.00000000000000111022302462515654e-01 1.70858975009092445285219241668528e-01 3.29141024990907471448053911444731e-01 6.15091271512330228787845953775104e-03
            4.19820677179887136176716921909247e-01 1.98257688837027712391858358387253e-01 3.81921633983085151431424719703500e-01 7.04483932597464636987671937617961e-03
            3.41717950018184946081589714594884e-01 2.24946792653554189511666550060909e-01 4.33335257328260836651168119715294e-01 7.68131669277919391924935865745283e-03
            2.67714629312019669615807515583583e-01 2.50235055699803310425011204642942e-01 4.82050314988177075470332511031302e-01 7.97658736156727291022328074632242e-03
            1.99727347669159671639249609143008e-01 2.73467530210110387756117233948316e-01 5.26805122120729940604633156908676e-01 7.87037473411890398100609900211566e-03
            1.39516911332385140642742271666066e-01 2.94042517084813437300283567310544e-01 5.66440571582801477568125392281218e-01 7.33286257650506406030599393375269e-03
            8.86426717314284795534717886766884e-02 3.11427157949986266327613293469767e-01 5.99930170318585309630066149111371e-01 6.36882405951634533725069786669337e-03
            4.84220481925909940201790959690697e-02 3.25171266974131012439386267942609e-01 6.26406684833277993540434636088321e-01 5.01846107291128995520068656333024e-03
            1.98959239325848247226247167418478e-02 3.34919155678224278460675122914836e-01 6.45184920389190841305548929085489e-01 3.35494136227543942110895791586245e-03
            3.79657807820771298068507348943967e-03 3.40420591140215833547699730843306e-01 6.55782830781576508982766426925082e-01 1.48090637586531384886634477027201e-03
            9.96203421921792342530466157768387e-01 1.01639949286151795235233485925619e-03 2.78017858534613951718150737235646e-03 5.26847616278835407030564294550423e-06
            9.80104076067414897721619126969017e-01 5.32642990043214609174881957187608e-03 1.45694940321529561866320534591068e-02 6.35755257563491016489440332826177e-05
            9.51577951807409005979820904030930e-01 1.29632906824082502761363855370291e-02 3.54587575101827437440427104320406e-02 2.38386937555439497706324947223777e-04
            9.11357328268571409424225748807658e-01 2.37309400038064645388491413768861e-02 6.49117317276221295063720617690706e-02 5.78264469721733831257537428172100e-04
            8.60483088667614692823804034560453e-01 3.73507182001074494737125064602878e-02 1.02166193132277857702483458979259e-01 1.10986759066613888818908506550542e-03
            8.00272652330840550405355315888301e-01 5.34699328447218805160368049200770e-02 1.46257414824437576017501783098851e-01 1.83361468996397496307448893304581e-03
            7.32285370687980496917646178189898e-01 7.16711227476720530438214495916327e-02 1.96043506564347436160744564404013e-01 2.72221223667818302410781328148914e-03
            6.58282049981815053918410285405116e-01 9.14828943183816495343663177663984e-02 2.50235055699803310425011204642942e-01 3.72224161568992899240537752802993e-03
            5.80179322820112863823283078090753e-01 1.12392136968734554502979960943776e-01 3.07428540211152567795949153151014e-01 4.75867209981673585511563473460228e-03
            5.00000000000000111022302462515654e-01 1.33857314656009807052328142162878e-01 3.66142685343990081925369395321468e-01 5.74186099082990824421512954245372e-03
            4.19820677179887136176716921909247e-01 1.55322492343285101235039746825350e-01 4.24856830476827762588243331265403e-01 6.57633915256549198002922196337749e-03
            3.41717950018184946081589714594884e-01 1.76231734993638033959229005631641e-01 4.82050314988177019959181279773475e-01 7.17048911587348306029277011930390e-03
            2.67714629312019669615807515583583e-01 1.96043506564347574938622642548580e-01 5.36241864123632838712296688754577e-01 7.44612351573775525792608931396899e-03
            1.99727347669159671639249609143008e-01 2.14244696467297712771937767683994e-01 5.86027955863542615588812623172998e-01 7.34697430479546928494549717925111e-03
            1.39516911332385140642742271666066e-01 2.30363911111912289531034048195579e-01 6.30119177555702680848526142654009e-01 6.84520810637237030887858324490480e-03
            8.86426717314284795534717886766884e-02 2.43983689308213236301980941789225e-01 6.67373638960358284144547269534087e-01 5.94528066296301071352869982433731e-03
            4.84220481925909940201790959690697e-02 2.54751338629611412400777226139326e-01 6.96826613177797593579043677891605e-01 4.68472033389439896505779259427982e-03
            1.98959239325848247226247167418478e-02 2.62388199411587585974103831176762e-01 7.17715876655827589303271452081390e-01 3.13182901900192767197239618326421e-03
            3.79657807820771298068507348943967e-03 2.66698229819158139086709979892476e-01 7.29505192102634203443756177875912e-01 1.38242224275846904701925410563490e-03
            9.96203421921792342530466157768387e-01 7.58280469779302004375287626913860e-04 3.03829760842835557357877185324924e-03 4.75669917446966059715708977084780e-06
            9.80104076067414897721619126969017e-01 3.97376011648256817487823155943261e-03 1.59221638161025358382261174483574e-02 5.73998328050988868877235749010879e-05
            9.51577951807409005979820904030930e-01 9.67120725421442666014382183448106e-03 3.87508409383765656253117981577816e-02 2.15230156507754168951004891674472e-04
            9.11357328268571409424225748807658e-01 1.77043657152262425003019785663128e-02 7.09383060162023515449192245796439e-02 5.22092164937257578911922184516925e-04
            8.60483088667614692823804034560453e-01 2.78653426554106437018631936552993e-02 1.11651568676974663474332771784248e-01 1.00205563984143912296953526919197e-03
            8.00272652330840550405355315888301e-01 3.98910134069573249493423361400346e-02 1.59836334262202117706408444064436e-01 1.65549832865352999390629040732392e-03
            7.32285370687980496917646178189898e-01 5.34699328447219221494002283634472e-02 2.14244696467297573994059689539426e-01 2.45777797959802635913151291902068e-03
            6.58282049981815053918410285405116e-01 6.82504198080745583254724806465674e-02 2.73467530210110387756117233948316e-01 3.36066503357930696296862471683653e-03
            5.80179322820112863823283078090753e-01 8.38496703498093631790055724195554e-02 3.35971006830077745242135733860778e-01 4.29641720857481135970035168725190e-03
            5.00000000000000111022302462515654e-01 9.98636738345798080640491889425903e-02 4.00136326165420053158072732912842e-01 5.18409965065592427679863618550371e-03
            4.19820677179887136176716921909247e-01 1.15877677319350308460244036723452e-01 4.64301645500762583118614656996215e-01 5.93751704505860074256862191077744e-03
            3.41717950018184946081589714594884e-01 1.31476927861085113313777128496440e-01 5.26805122120729940604633156908676e-01 6.47395159513589144723022528182810e-03
            2.67714629312019669615807515583583e-01 1.46257414824437714795379861243418e-01 5.86027955863542615588812623172998e-01 6.72281101516141853940977668457890e-03
            1.99727347669159671639249609143008e-01 1.59836334262202284239862137837918e-01 6.40436318068638099632039484276902e-01 6.63329310613687030129259625255145e-03
            1.39516911332385140642742271666066e-01 1.71862005013749069570749838931079e-01 6.88621083653865873053234736289596e-01 6.18026821087897058198956656838163e-03
            8.86426717314284795534717886766884e-02 1.82022981953933460363970198159222e-01 7.29334346314638004571406781906262e-01 5.36775924341268643319446951522877e-03
            4.84220481925909940201790959690697e-02 1.90056140414945257122170119146176e-01 7.61521811392463776613226400513668e-01 4.22964907808616973061699439995209e-03
            1.98959239325848247226247167418478e-02 1.95753587552677132954670469189296e-01 7.84350488514737986811553582811030e-01 2.82760480430497684184443230037687e-03
            3.79657807820771298068507348943967e-03 1.98969067199380372779060621724057e-01 7.97234354722411997506981151673244e-01 1.24813447716492381157904478783394e-03
            9.96203421921792342530466157768387e-01 5.29686847103782651458026009549940e-04 3.26689123110387481807581622206271e-03 4.12172922919725862641714825285177e-06
            9.80104076067414897721619126969017e-01 2.77581785517834727511354842022229e-03 1.71201060774067541359055866223571e-02 4.97375511770070048277875285336336e-05
            9.51577951807409005979820904030930e-01 6.75569460421819763251694013206361e-03 4.16663535883727981223856318138132e-02 1.86499165607128966843167305889040e-04
            9.11357328268571409424225748807658e-01 1.23671517722194531513357773633288e-02 7.62755199592091426286089017594350e-02 4.52398282428002223611029908312275e-04
            8.60483088667614692823804034560453e-01 1.94649685477286393808782349879039e-02 1.20051942784656667795317730451643e-01 8.68291617469567973859601028863153e-04
            8.00272652330840550405355315888301e-01 2.78653426554105708434772026294013e-02 1.71862005013748875281720529528684e-01 1.43450649280532616623717867554433e-03
            7.32285370687980496917646178189898e-01 3.73507182001073800847734673880041e-02 2.30363911111912122997580354422098e-01 2.12969014138171238229024773147557e-03
            6.58282049981815053918410285405116e-01 4.76754329333715295979878590060252e-02 2.94042517084813437300283567310544e-01 2.91205114941694639951741407912778e-03
            5.80179322820112863823283078090753e-01 5.85720841936082023404530616517150e-02 3.61248592986278926897369956350303e-01 3.72289012608902026776891425186022e-03
            5.00000000000000111022302462515654e-01 6.97584556661925564435833280185761e-02 4.30241544333807346411902017280227e-01 4.49207618002501595649844645663507e-03
            4.19820677179887136176716921909247e-01 8.09448271387769452411831139215792e-02 4.99234495681335932459887771983631e-01 5.14492017591243206570128521093466e-03
            3.41717950018184946081589714594884e-01 9.18414783990136179836483165672689e-02 5.66440571582801477568125392281218e-01 5.60974628399848063847743517840172e-03
            2.67714629312019669615807515583583e-01 1.02166193132277746680180996463605e-01 6.30119177555702680848526142654009e-01 5.82538555565678423803221974708322e-03
            1.99727347669159671639249609143008e-01 1.11651568676974524696454693639680e-01 6.88621083653865762030932273773942e-01 5.74781735791507226301932931278316e-03
            1.39516911332385140642742271666066e-01 1.20051942784656529017439652307075e-01 7.40431145882958441362120538542513e-01 5.35526658187262909194270577017960e-03
            8.86426717314284795534717886766884e-02 1.27149759560165720451152537862072e-01 7.84207568708405799995375673461240e-01 4.65121912430683723260749218297860e-03
            4.84220481925909940201790959690697e-02 1.32761216728166930867160999696353e-01 8.18816735079242130623811135592405e-01 3.66503484768693904571890129773237e-03
            1.98959239325848247226247167418478e-02 1.36741093477206809847501745025511e-01 8.43362982590208254407571075716987e-01 2.45014892534628839215748996593902e-03
            3.79657807820771298068507348943967e-03 1.38987224485281363195454673586937e-01 8.57216197436511007090587099810364e-01 1.08152148534242293666240986027560e-03
            9.96203421921792342530466157768387e-01 3.36538824289303909690490579365019e-04 3.46003925391835361405346027652286e-03 3.38001644175229110886619562914568e-06
            9.80104076067414897721619126969017e-01 1.76362785394960789112084320606755e-03 1.81322960786354930862174228423100e-02 4.07871869796554468393795511715894e-05
            9.51577951807409005979820904030930e-01 4.29225972249925279144999734626253e-03 4.41297884700917394940056226460001e-02 1.52938296300446016640073310099979e-04
            9.11357328268571409424225748807658e-01 7.85752325168580408043528251482712e-03 8.07851484797427882300624446543225e-02 3.70988375945537768307858073058014e-04
            8.60483088667614692823804034560453e-01 1.23671517722194444777183974792933e-02 1.27149759560165859229030616006639e-01 7.12040937209845585000034251521583e-04
            8.00272652330840550405355315888301e-01 1.77043657152261835197037953548715e-02 1.82022981953933266074940888756828e-01 1.17636439995519401918722124378291e-03
            7.32285370687980496917646178189898e-01 2.37309400038064055582509581654449e-02 2.43983689308213097524102863644657e-01 1.74644846699692035500717235407819e-03
            6.58282049981815053918410285405116e-01 3.02907920681986519984008054962032e-02 3.11427157949986321838764524727594e-01 2.38802216665016118005149259317932e-03
            5.80179322820112863823283078090753e-01 3.72140264733227393856296316698717e-02 3.82606650706564410668875098053832e-01 3.05294917188649907180408860085663e-03
            5.00000000000000111022302462515654e-01 4.43213358657142328378419904311158e-02 4.55678664134285649200961643146002e-01 3.68371877476421534514106070901107e-03
            4.19820677179887136176716921909247e-01 5.14286452581057401678421570068167e-02 5.28750677562007109777653113269480e-01 4.21908228781781111799142891527481e-03
            3.41717950018184946081589714594884e-01 5.83518796632298275550709831804852e-02 5.99930170318585198607763686595717e-01 4.60026207924058710974080099731509e-03
            2.67714629312019669615807515583583e-01 6.49117317276220601174330226967868e-02 6.67373638960358284144547269534087e-01 4.77709666568781627510320575424885e-03
            1.99727347669159671639249609143008e-01 7.09383060162022682781923776929034e-02 7.29334346314638004571406781906262e-01 4.71348700839474381296678728858751e-03
            1.39516911332385140642742271666066e-01 7.62755199592090593618820548726944e-02 7.84207568708405799995375673461240e-01 4.39157646952499647330370180497994e-03
            8.86426717314284795534717886766884e-02 8.07851484797426910855477899531252e-02 8.30572179788828801605404805741273e-01 3.81422365229256918328637482318300e-03
            4.84220481925909940201790959690697e-02 8.43504120089292319661922192608472e-02 8.67227539798479774013628684770083e-01 3.00550505768900364303441996582933e-03
            1.98959239325848247226247167418478e-02 8.68790438774788842390961463024723e-02 8.93225032189936207771552290068939e-01 2.00924010091392199472970503393299e-03
            3.79657807820771298068507348943967e-03 8.83061329071391787337574896810111e-02 9.07897289014653163796708668087376e-01 8.86899696532879788443692881116931e-04
            9.96203421921792342530466157768387e-01 1.83838086669908389340563625857783e-04 3.61273999153774926992865879071815e-03 2.55078844149308508511145646147078e-06
            9.80104076067414897721619126969017e-01 9.63401387499757664613364571692955e-04 1.89325225450853451558685947020422e-02 3.07807630233850352749480627778667e-05
            9.51577951807409005979820904030930e-01 2.34469475116560457580994203397040e-03 4.60773534414253907454117609177047e-02 1.15417556449097407950295235412597e-04
            9.11357328268571409424225748807658e-01 4.29225972249926146506737723029801e-03 8.43504120089293291107068739620445e-02 2.79972857409999792682037123014993e-04
            8.60483088667614692823804034560453e-01 6.75569460421820630613432001609908e-03 1.32761216728167097400614693469834e-01 5.37354129426416936288202208515941e-04
            8.00272652330840550405355315888301e-01 9.67120725421441278235601402002430e-03 1.90056140414945035077565194114868e-01 8.87763940235165513459314468036609e-04
            7.32285370687980496917646178189898e-01 1.29632906824082416025190056529937e-02 2.54751338629611245867323532365845e-01 1.31798783823949806036368581629858e-03
            6.58282049981815053918410285405116e-01 1.65466830440539509894382064203455e-02 3.25171266974131012439386267942609e-01 1.80216263609729414330029584334625e-03
            5.80179322820112863823283078090753e-01 2.03285770626506802349897640169729e-02 3.99492100117236459411174109845888e-01 2.30396141389086189094070888927490e-03
            5.00000000000000111022302462515654e-01 2.42110240962954900711956440773065e-02 4.75788975903704391967607989499811e-01 2.77998271141790403992999358706584e-03
            4.19820677179887136176716921909247e-01 2.80934711299403137851893319520968e-02 5.52085851690172546568646794185042e-01 3.18400413694276945147931812130082e-03
            3.41717950018184946081589714594884e-01 3.18753651485370395612939375951100e-02 6.26406684833277993540434636088321e-01 3.47166812404097781252776400151561e-03
            2.67714629312019669615807515583583e-01 3.54587575101827437440427104320406e-02 6.96826613177797593579043677891605e-01 3.60511943321033690584842545945321e-03
            1.99727347669159671639249609143008e-01 3.87508409383765656253117981577816e-02 7.61521811392463776613226400513668e-01 3.55711529435875483534656993356293e-03
            1.39516911332385140642742271666066e-01 4.16663535883727981223856318138132e-02 8.18816735079242130623811135592405e-01 3.31417988386765441019554678803161e-03
            8.86426717314284795534717886766884e-02 4.41297884700917394940056226460001e-02 8.67227539798479774013628684770083e-01 2.87847049657945089698452001414353e-03
            4.84220481925909940201790959690697e-02 4.60773534414253907454117609177047e-02 9.05500598365983622173303047020454e-01 2.26815688447587785955827932582451e-03
            1.98959239325848247226247167418478e-02 4.74586468050912438365607215473574e-02 9.32645429262323855112981618731283e-01 1.51630813456594287259859843430831e-03
            3.79657807820771298068507348943967e-03 4.82382101059210879889782574991841e-02 9.47965211815871233724806188547518e-01 6.69314346147665824131312017897244e-04
            9.96203421921792342530466157768387e-01 7.55364286481397320671682393822266e-05 3.72104164955951773746667399223043e-03 1.65562124269201559833522997139399e-06
            9.80104076067414897721619126969017e-01 3.95847789131206025028258954634452e-04 1.95000761434538961991425054520732e-02 1.99786404465418393505173055846313e-05
            9.51577951807409005979820904030930e-01 9.63401387499746931011856965199058e-04 4.74586468050912438365607215473574e-02 7.49132131572929292053983352239754e-05
            9.11357328268571409424225748807658e-01 1.76362785394959141124782142640015e-03 8.68790438774789952613986088181264e-02 1.81719895921221541966461554551415e-04
            8.60483088667614692823804034560453e-01 2.77581785517831995321880178551055e-03 1.36741093477206976380955438798992e-01 3.48776439886130907534061984520690e-04
            8.00272652330840550405355315888301e-01 3.97376011648251786789742823202687e-03 1.95753587552676938665641159786901e-01 5.76214324183219137737177462099680e-04
            7.32285370687980496917646178189898e-01 5.32642990043208364170368440682068e-03 2.62388199411587419440650137403281e-01 8.55456543201901743397885802266956e-04
            6.58282049981815053918410285405116e-01 6.79879433996063119172159616709905e-03 3.34919155678224333971826354172663e-01 1.16971627069238051817301471402288e-03
            5.80179322820112863823283078090753e-01 8.35272025849728358415369200429268e-03 4.11467956921389876878691893580253e-01 1.49541506348823652512713966444835e-03
            5.00000000000000111022302462515654e-01 9.94796196629241062658888239411681e-03 4.90052038033707504371960794742336e-01 1.80438265928707609539460854364279e-03
            4.19820677179887136176716921909247e-01 1.15432036740875411384710247375551e-02 5.68636119146025298398683389677899e-01 2.06661783478055520457683513768643e-03
            3.41717950018184946081589714594884e-01 1.30971295926241935309031205747488e-02 6.45184920389190841305548929085489e-01 2.25332975492650768728486454506310e-03
            2.67714629312019669615807515583583e-01 1.45694940321527393461975563582200e-02 7.17715876655827700325573914597044e-01 2.33994800155637016486620538557872e-03
            1.99727347669159671639249609143008e-01 1.59221638161022999158333846025926e-02 7.84350488514737986811553582811030e-01 2.30879031292685176629531085268354e-03
            1.39516911332385140642742271666066e-01 1.71201060774065078051719979157497e-02 8.43362982590208365429873538232641e-01 2.15111003663716457076637844636480e-03
            8.86426717314284795534717886766884e-02 1.81322960786352363471429782748601e-02 8.93225032189936318793854752584593e-01 1.86830739197235500513583073001200e-03
            4.84220481925909940201790959690697e-02 1.89325225450850780084532942737496e-02 9.32645429262323966135284081246937e-01 1.47217568443203366407301402318808e-03
            1.98959239325848247226247167418478e-02 1.95000761434536221128333011165523e-02 9.60603999923961504592284654791001e-01 9.84178819857210599245100901555361e-04
            3.79657807820771298068507348943967e-03 1.98203875039366858579192864908691e-02 9.76383034417855677489228582999203e-01 4.34426874253811506725042601573250e-04
            9.96203421921792342530466157768387e-01 1.44140051039273714401185363942481e-05 3.78216407310373028613503798567308e-03 7.18998236483508931737808258372047e-07
            9.80104076067414897721619126969017e-01 7.55364286481405858763790717169684e-05 1.98203875039369634136754427800042e-02 8.67626415873056088695104703933936e-06
            9.51577951807409005979820904030930e-01 1.83838086669908389340563625857783e-04 4.82382101059210879889782574991841e-02 3.25330859259979267186606299855356e-05
            9.11357328268571409424225748807658e-01 3.36538824289304560211794070667679e-04 8.83061329071392897560599521966651e-02 7.89167723463609944046376343251836e-05
            8.60483088667614692823804034560453e-01 5.29686847103783301979329500852600e-04 1.38987224485281529728908367360418e-01 1.51465588105995178089605013305174e-04
            8.00272652330840550405355315888301e-01 7.58280469779300920173115141409426e-04 1.98969067199380150734455696692748e-01 2.50236631568359548394991920261532e-04
            7.32285370687980496917646178189898e-01 1.01639949286151730183103136795353e-03 2.66698229819157972553256286118994e-01 3.71505106415733377285548488799805e-04
            6.58282049981815053918410285405116e-01 1.29735887796911990646475665300841e-03 3.40420591140215833547699730843306e-01 5.07980880002720865291909735361742e-04
            5.80179322820112863823283078090753e-01 1.59388197975947657446249561274954e-03 4.18226795200127676732648751567467e-01 6.49424376622914070397718511173935e-04
            5.00000000000000111022302462515654e-01 1.89828903910385605666166775051806e-03 4.98101710960896060242930616368540e-01 7.83601899103060104898044180288252e-04
            4.19820677179887136176716921909247e-01 2.20269609844823662306301237379103e-03 5.77976626721664610286666174943093e-01 8.97484606005987404012125274732625e-04
            3.41717950018184946081589714594884e-01 2.49921920023859307422031683643127e-03 6.55782830781576508982766426925082e-01 9.78569299686957638426121341979069e-04
            2.67714629312019669615807515583583e-01 2.78017858534619546201360762438526e-03 7.29505192102634203443756177875912e-01 1.01618561250552304209282539204651e-03
            1.99727347669159671639249609143008e-01 3.03829760842841108473000311107626e-03 7.97234354722411886484678689157590e-01 1.00265454477103314705199732514984e-03
            1.39516911332385140642742271666066e-01 3.26689123110393032922704747988973e-03 8.57216197436511007090587099810364e-01 9.34177626465626130200514687373925e-04
            8.86426717314284795534717886766884e-02 3.46003925391840912520469153434988e-03 9.07897289014653163796708668087376e-01 8.11362940628271604379118109307001e-04
            4.84220481925909940201790959690697e-02 3.61273999153780478107989004854517e-03 9.47965211815871233724806188547518e-01 6.39332048663161117176001457806933e-04
            1.98959239325848247226247167418478e-02 3.72104164955957324861790525005745e-03 9.76383034417855566466926120483549e-01 4.27406231337773394058504594639203e-04
            3.79657807820771298068507348943967e-03 3.78216407310378579728626924350010e-03 9.92421257848688509461965168156894e-01 1.88661602312889872942389124510498e-04
            ];
        
    case  36
        
        % ALG. DEG.:   36
        % PTS CARD.:  361
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.8e-15
        
        xyw_bar=[
            9.96203421921792342530466157768387e-01 3.78216407310378579728626924350010e-03 1.44140051038716722475729881125517e-05 7.18998236483455251024776017088058e-07
            9.80104076067414897721619126969017e-01 1.98203875039369079025242115221772e-02 7.55364286481943758566615088057006e-05 8.67626415872991205971345024527963e-06
            9.51577951807409005979820904030930e-01 4.82382101059210879889782574991841e-02 1.83838086669906031200838469885639e-04 3.25330859259954940400361156349618e-05
            9.11357328268571409424225748807658e-01 8.83061329071393452672111834544921e-02 3.36538824289245308563067737850361e-04 7.89167723463550990553247443948237e-05
            8.60483088667614692823804034560453e-01 1.38987224485281529728908367360418e-01 5.29686847103777447287598079128657e-04 1.51465588105983848176902539783839e-04
            8.00272652330840550405355315888301e-01 1.98969067199380095223304465434921e-01 7.58280469779354371340218676778022e-04 2.50236631568340845907516545310045e-04
            7.32285370687980496917646178189898e-01 2.66698229819158028064407517376821e-01 1.01639949286147501794630443328060e-03 3.71505106415705621709932859886294e-04
            6.58282049981815053918410285405116e-01 3.40420591140215833547699730843306e-01 1.29735887796911253388998375157826e-03 5.07980880002682918215872742706551e-04
            5.80179322820112863823283078090753e-01 4.18226795200127676732648751567467e-01 1.59388197975945944406817034177948e-03 6.49424376622865606560608409125734e-04
            5.00000000000000111022302462515654e-01 4.98101710960896060242930616368540e-01 1.89828903910382873476692111580633e-03 7.83601899103001557980729963048816e-04
            4.19820677179887136176716921909247e-01 5.77976626721664610286666174943093e-01 2.20269609844825353661690314766020e-03 8.97484606005920400317865670558604e-04
            3.41717950018184946081589714594884e-01 6.55782830781576508982766426925082e-01 2.49921920023854493564385848003440e-03 9.78569299686884563199695818980217e-04
            2.67714629312019669615807515583583e-01 7.29505192102634203443756177875912e-01 2.78017858534618245158753779833205e-03 1.01618561250544714794075140673613e-03
            1.99727347669159671639249609143008e-01 7.97234354722411997506981151673244e-01 3.03829760842833085376923918374814e-03 1.00265454477095833710209582534389e-03
            1.39516911332385140642742271666066e-01 8.57216197436511007090587099810364e-01 3.26689123110390777782185978139751e-03 9.34177626465556307580606620888375e-04
            8.86426717314284795534717886766884e-02 9.07897289014653163796708668087376e-01 3.46003925391835664981954323593527e-03 8.11362940628210997477676169609140e-04
            4.84220481925909940201790959690697e-02 9.47965211815871233724806188547518e-01 3.61273999153777225501471548341215e-03 6.39332048663113303860194847061393e-04
            1.98959239325848247226247167418478e-02 9.76383034417855566466926120483549e-01 3.72104164955955329929793151677586e-03 4.27406231337741464304524896533621e-04
            3.79657807820771298068507348943967e-03 9.92421257848688620484267630672548e-01 3.78216407310372204619852709583938e-03 1.88661602312875778314146812952856e-04
            9.96203421921792342530466157768387e-01 3.72104164955957238125616726165390e-03 7.55364286480850882776749699587526e-05 1.65562124269206578503735478868908e-06
            9.80104076067414897721619126969017e-01 1.95000761434538337490973702870178e-02 3.95847789131268529283502743965073e-04 1.99786404465424458261075396636741e-05
            9.51577951807409005979820904030930e-01 4.74586468050912368976668176401290e-02 9.63401387499757122512278328940738e-04 7.49132131572951924774333987144814e-05
            9.11357328268571409424225748807658e-01 8.68790438774790368947620322614966e-02 1.76362785394955368101221893084585e-03 1.81719895921227071397541230624029e-04
            8.60483088667614692823804034560453e-01 1.36741093477206948625379823170078e-01 2.77581785517835855081614226946840e-03 3.48776439886141532715352342464143e-04
            8.00272652330840550405355315888301e-01 1.95753587552676827643338697271247e-01 3.97376011648262195130598684045253e-03 5.76214324183236701812371727271511e-04
            7.32285370687980496917646178189898e-01 2.62388199411587363929498906145454e-01 5.32642990043213915285491566464771e-03 8.55456543201927655829808205822928e-04
            6.58282049981815053918410285405116e-01 3.34919155678224222949523891657009e-01 6.79879433996072313206582293787505e-03 1.16971627069241608000427223856832e-03
            5.80179322820112863823283078090753e-01 4.11467956921389710345238199806772e-01 8.35272025849742583147872210247442e-03 1.49541506348828184477794955853369e-03
            5.00000000000000111022302462515654e-01 4.90052038033707337838507100968855e-01 9.94796196629255113919043651549146e-03 1.80438265928713073918410181306626e-03
            4.19820677179887136176716921909247e-01 5.68636119146025187376380927162245e-01 1.15432036740876764469021509285085e-02 2.06661783478061808830283929694360e-03
            3.41717950018184946081589714594884e-01 6.45184920389190730283246466569835e-01 1.30971295926243236351638188352808e-02 2.25332975492657577518129663474156e-03
            2.67714629312019669615807515583583e-01 7.17715876655827478280968989565736e-01 1.45694940321529076143747261085082e-02 2.33994800155644128852872043466959e-03
            1.99727347669159671639249609143008e-01 7.84350488514737764766948657779722e-01 1.59221638161025635938017330772709e-02 2.30879031292692202259608791337087e-03
            1.39516911332385140642742271666066e-01 8.43362982590208143385268613201333e-01 1.71201060774067714831403463904280e-02 2.15111003663722962289672757663084e-03
            8.86426717314284795534717886766884e-02 8.93225032189935985726947365037631e-01 1.81322960786355347195808462856803e-02 1.86830739197241160048923447334346e-03
            4.84220481925909940201790959690697e-02 9.32645429262323633068376693699975e-01 1.89325225450853729114442103309557e-02 1.47217568443207833320252042597076e-03
            1.98959239325848247226247167418478e-02 9.60603999923961171525377267244039e-01 1.95000761434539482408467847562861e-02 9.84178819857240523225061501477740e-04
            3.79657807820771298068507348943967e-03 9.76383034417855344422321195452241e-01 1.98203875039369981081449623161461e-02 4.34426874253824733991546924727345e-04
            9.96203421921792342530466157768387e-01 3.61273999153780478107989004854517e-03 1.83838086669852688453952183067486e-04 2.55078844149308762621029822437180e-06
            9.80104076067414897721619126969017e-01 1.89325225450852896447173634442152e-02 9.63401387499812633663509586767759e-04 3.07807630233850623800023749154775e-05
            9.51577951807409005979820904030930e-01 4.60773534414253907454117609177047e-02 2.34469475116560327476733505136508e-03 1.15417556449097516370512483963040e-04
            9.11357328268571409424225748807658e-01 8.43504120089293846218581052198715e-02 4.29225972249920595391614597247099e-03 2.79972857410000063732580244391102e-04
            8.60483088667614692823804034560453e-01 1.32761216728167097400614693469834e-01 6.75569460421820977558127196971327e-03 5.37354129426417478389288451268158e-04
            8.00272652330840550405355315888301e-01 1.90056140414944979566413962857041e-01 9.67120725421447002823072125465842e-03 8.87763940235166380821052456440157e-04
            7.32285370687980496917646178189898e-01 2.54751338629611245867323532365845e-01 1.29632906824082572150302894442575e-02 1.31798783823949936140629279890391e-03
            6.58282049981815053918410285405116e-01 3.25171266974131012439386267942609e-01 1.65466830440539336422034466522746e-02 1.80216263609729609486420631725423e-03
            5.80179322820112863823283078090753e-01 3.99492100117236459411174109845888e-01 2.03285770626506767655428120633587e-02 2.30396141389086405934505386028377e-03
            5.00000000000000111022302462515654e-01 4.75788975903704391967607989499811e-01 2.42110240962954970100895479845349e-02 2.77998271141790664201520755227648e-03
            4.19820677179887136176716921909247e-01 5.52085851690172546568646794185042e-01 2.80934711299403172546362839057110e-02 3.18400413694277292092627007491501e-03
            3.41717950018184946081589714594884e-01 6.26406684833277993540434636088321e-01 3.18753651485370603779756493167952e-02 3.47166812404098128197471595512980e-03
            2.67714629312019669615807515583583e-01 6.96826613177797593579043677891605e-01 3.54587575101827923163000377826393e-02 3.60511943321034037529537741306740e-03
            1.99727347669159671639249609143008e-01 7.61521811392463776613226400513668e-01 3.87508409383765517475239903433248e-02 3.55711529435875873847439088137889e-03
            1.39516911332385140642742271666066e-01 8.18816735079242130623811135592405e-01 4.16663535883727842445978239993565e-02 3.31417988386765787964249874164580e-03
            8.86426717314284795534717886766884e-02 8.67227539798479774013628684770083e-01 4.41297884700917464328995265532285e-02 2.87847049657945393275060297355594e-03
            4.84220481925909940201790959690697e-02 9.05500598365983622173303047020454e-01 4.60773534414253838065178570104763e-02 2.26815688447588002796262429683338e-03
            1.98959239325848247226247167418478e-02 9.32645429262323855112981618731283e-01 4.74586468050912646532424332690425e-02 1.51630813456594439048163991401452e-03
            3.79657807820771298068507348943967e-03 9.47965211815871233724806188547518e-01 4.82382101059211088056599692208692e-02 6.69314346147666474652615509199904e-04
            9.96203421921792342530466157768387e-01 3.46003925391840869152382254014810e-03 3.36538824289248778010019691464549e-04 3.38001644175229746161330003639822e-06
            9.80104076067414897721619126969017e-01 1.81322960786354375750661915844830e-02 1.76362785394966470331468144649989e-03 4.07871869796555281545424875844219e-05
            9.51577951807409005979820904030930e-01 4.41297884700917325551117187387717e-02 4.29225972249926146506737723029801e-03 1.52938296300446314795670743613698e-04
            9.11357328268571409424225748807658e-01 8.07851484797428298634258680976927e-02 7.85752325168576071234838309464976e-03 3.70988375945538527249378812911118e-04
            8.60483088667614692823804034560453e-01 1.27149759560165859229030616006639e-01 1.23671517722194479471653494329075e-02 7.12040937209846994462858482677348e-04
            8.00272652330840550405355315888301e-01 1.82022981953933182808214041870087e-01 1.77043657152262667864306422416121e-02 1.17636439995519640443200071189267e-03
            7.32285370687980496917646178189898e-01 2.43983689308213069768527248015744e-01 2.37309400038064333138265737943584e-02 1.74644846699692360761368981059150e-03
            6.58282049981815053918410285405116e-01 3.11427157949986266327613293469767e-01 3.02907920681986797539764211251168e-02 2.38802216665016595054105152939883e-03
            5.80179322820112863823283078090753e-01 3.82606650706564355157723866796005e-01 3.72140264733227810189930551132420e-02 3.05294917188650514333625451968146e-03
            5.00000000000000111022302462515654e-01 4.55678664134285593689810411888175e-01 4.43213358657142952878871255961712e-02 3.68371877476422228403496461623945e-03
            4.19820677179887136176716921909247e-01 5.28750677562007109777653113269480e-01 5.14286452581057540456299648212735e-02 4.21908228781781892424707081090673e-03
            3.41717950018184946081589714594884e-01 5.99930170318585198607763686595717e-01 5.83518796632298553106465988093987e-02 4.60026207924059578335818088135056e-03
            2.67714629312019669615807515583583e-01 6.67373638960358284144547269534087e-01 6.49117317276221017507964461401571e-02 4.77709666568782581608232362668787e-03
            1.99727347669159671639249609143008e-01 7.29334346314638004571406781906262e-01 7.09383060162023237893436089507304e-02 4.71348700839475248658416717262298e-03
            1.39516911332385140642742271666066e-01 7.84207568708405799995375673461240e-01 7.62755199592091148730332861305214e-02 4.39157646952500514692108168901541e-03
            8.86426717314284795534717886766884e-02 8.30572179788828690583102343225619e-01 8.07851484797428298634258680976927e-02 3.81422365229257698954201671881492e-03
            4.84220481925909940201790959690697e-02 8.67227539798479662991326222254429e-01 8.43504120089293429884946817765012e-02 3.00550505768900971456658588465416e-03
            1.98959239325848247226247167418478e-02 8.93225032189936096749249827553285e-01 8.68790438774790230169742244470399e-02 2.00924010091392589785752598174895e-03
            3.79657807820771298068507348943967e-03 9.07897289014653052774406205571722e-01 8.83061329071392897560599521966651e-02 8.86899696532881523167168857924025e-04
            9.96203421921792342530466157768387e-01 3.26689123110392946186530949148619e-03 5.29686847103728007668532740126466e-04 4.12172922919727556707609333885856e-06
            9.80104076067414897721619126969017e-01 1.71201060774066951553074034109159e-02 2.77581785517840712307346962006704e-03 4.97375511770072148919584476001177e-05
            9.51577951807409005979820904030930e-01 4.16663535883727911834917279065849e-02 6.75569460421820283668736806248489e-03 1.86499165607129752889742357879754e-04
            9.11357328268571409424225748807658e-01 7.62755199592091842619723252028052e-02 1.23671517722194063138019259895373e-02 4.52398282428004120964831757945035e-04
            8.60483088667614692823804034560453e-01 1.20051942784656640039742114822729e-01 1.94649685477286671364538506168174e-02 8.68291617469571551726770231027785e-04
            8.00272652330840550405355315888301e-01 1.71862005013748792014993682641943e-01 2.78653426554106575796510014697560e-02 1.43450649280533223776934459436916e-03
            7.32285370687980496917646178189898e-01 2.30363911111912067486429123164271e-01 3.73507182001074355959246986458311e-02 2.12969014138172148958849660971282e-03
            6.58282049981815053918410285405116e-01 2.94042517084813381789132336052717e-01 4.76754329333715642924573785421671e-02 2.91205114941695854258174591677744e-03
            5.80179322820112863823283078090753e-01 3.61248592986278871386218725092476e-01 5.85720841936082647904981968167704e-02 3.72289012608903588028019804312407e-03
            5.00000000000000111022302462515654e-01 4.30241544333807235389599554764573e-01 6.97584556661926535880979827197734e-02 4.49207618002503503845668220151310e-03
            4.19820677179887136176716921909247e-01 4.99234495681335821437585309467977e-01 8.09448271387770423856977686227765e-02 5.14492017591245461710647290942688e-03
            3.41717950018184946081589714594884e-01 5.66440571582801366545822929765563e-01 9.18414783990136873725873556395527e-02 5.60974628399850405724436086529749e-03
            2.67714629312019669615807515583583e-01 6.30119177555702569826223680138355e-01 1.02166193132277816069120035535889e-01 5.82538555565680852416088342238254e-03
            1.99727347669159671639249609143008e-01 6.88621083653865651008629811258288e-01 1.11651568676974677352120579598704e-01 5.74781735791509654914799298808248e-03
            1.39516911332385140642742271666066e-01 7.40431145882958219317515613511205e-01 1.20051942784656695550893346080557e-01 5.35526658187265251070963145707537e-03
            8.86426717314284795534717886766884e-02 7.84207568708405688973073210945586e-01 1.27149759560165831473455000377726e-01 4.65121912430685718192746591626019e-03
            4.84220481925909940201790959690697e-02 8.18816735079241908579206210561097e-01 1.32761216728167097400614693469834e-01 3.66503484768695465823018508899622e-03
            1.98959239325848247226247167418478e-02 8.43362982590208143385268613201333e-01 1.36741093477206976380955438798992e-01 2.45014892534629836681747683257981e-03
            3.79657807820771298068507348943967e-03 8.57216197436510785045982174779056e-01 1.38987224485281557484483982989332e-01 1.08152148534242749031153429939422e-03
            9.96203421921792342530466157768387e-01 3.03829760842841195209174109947980e-03 7.58280469779245517442101132132848e-04 4.75669917446964873869582821064306e-06
            9.80104076067414897721619126969017e-01 1.59221638161024837965218381441446e-02 3.97376011648261848185903488683834e-03 5.73998328050987378099248581442282e-05
            9.51577951807409005979820904030930e-01 3.87508409383765795030996059722384e-02 9.67120725421441451707948999683140e-03 2.15230156507753626849918648922255e-04
            9.11357328268571409424225748807658e-01 7.09383060162024209338582636519277e-02 1.77043657152261696419159875404148e-02 5.22092164937256277869315201911604e-04
            8.60483088667614692823804034560453e-01 1.11651568676974691229908387413161e-01 2.78653426554106159462875780263857e-02 1.00205563984143652088432130398132e-03
            8.00272652330840550405355315888301e-01 1.59836334262202117706408444064436e-01 3.98910134069573318882362400472630e-02 1.65549832865352565709760046530619e-03
            7.32285370687980496917646178189898e-01 2.14244696467297629505210920797253e-01 5.34699328447218735771429010128486e-02 2.45777797959801985391847800599407e-03
            6.58282049981815053918410285405116e-01 2.73467530210110443267268465206143e-01 6.82504198080745028143212493887404e-02 3.36066503357929828935124483280106e-03
            5.80179322820112863823283078090753e-01 3.35971006830077856264438196376432e-01 8.38496703498092799122787255328149e-02 4.29641720857480008399775783800578e-03
            5.00000000000000111022302462515654e-01 4.00136326165420164180375195428496e-01 9.98636738345797247973223420558497e-02 5.18409965065591126637256635945050e-03
            4.19820677179887136176716921909247e-01 4.64301645500762694140917119511869e-01 1.15877677319350169682365958578885e-01 5.93751704505858513005733811951359e-03
            3.41717950018184946081589714594884e-01 5.26805122120730051626935619424330e-01 1.31476927861085002291474665980786e-01 6.47395159513587496735720350216070e-03
            2.67714629312019669615807515583583e-01 5.86027955863542837633417548204307e-01 1.46257414824437548261926167469937e-01 6.72281101516140119217501691650796e-03
            1.99727347669159671639249609143008e-01 6.40436318068638210654341946792556e-01 1.59836334262202117706408444064436e-01 6.63329310613685382141957447288405e-03
            1.39516911332385140642742271666066e-01 6.88621083653865984075537198805250e-01 1.71862005013748930792871760786511e-01 6.18026821087895496947828277711778e-03
            8.86426717314284795534717886766884e-02 7.29334346314638226616011706937570e-01 1.82022981953933293830516504385741e-01 5.36775924341267255540666170077202e-03
            4.84220481925909940201790959690697e-02 7.61521811392463998657831325544976e-01 1.90056140414945007321989578485955e-01 4.22964907808615932227613853910952e-03
            1.98959239325848247226247167418478e-02 7.84350488514738208856158507842338e-01 1.95753587552676910910065544157987e-01 2.82760480430496946926965939894671e-03
            3.79657807820771298068507348943967e-03 7.97234354722412219551586076704552e-01 1.98969067199380122978880081063835e-01 1.24813447716492055897252733132063e-03
            9.96203421921792342530466157768387e-01 2.78017858534619589569447661858703e-03 1.01639949286146157383936561302562e-03 5.26847616278834051777848687669881e-06
            9.80104076067414897721619126969017e-01 1.45694940321529041449277741548940e-02 5.32642990043219813345309887608892e-03 6.35755257563489390186181604569526e-05
            9.51577951807409005979820904030930e-01 3.54587575101827506829366143392690e-02 1.29632906824082433372424816298007e-02 2.38386937555438874290075768058728e-04
            9.11357328268571409424225748807658e-01 6.49117317276221988953111008413543e-02 2.37309400038063916804631503509881e-02 5.78264469721732313374495948465892e-04
            8.60483088667614692823804034560453e-01 1.02166193132277885458059074608173e-01 3.73507182001074217181368908313743e-02 1.10986759066613606926343660319390e-03
            8.00272652330840550405355315888301e-01 1.46257414824437548261926167469937e-01 5.34699328447219013327185166417621e-02 1.83361468996397040942536449392719e-03
            7.32285370687980496917646178189898e-01 1.96043506564347491671895795661840e-01 7.16711227476720114104580261482624e-02 2.72221223667817608521390937426077e-03
            6.58282049981815053918410285405116e-01 2.50235055699803365936162435900769e-01 9.14828943183815801454272786941146e-02 3.72224161568991945142625965559091e-03
            5.80179322820112863823283078090753e-01 3.07428540211152623307100384408841e-01 1.12392136968734512869616537500406e-01 4.75867209981672371205130289695262e-03
            5.00000000000000111022302462515654e-01 3.66142685343990192947671857837122e-01 1.33857314656009696030025679647224e-01 5.74186099082989349906558373959342e-03
            4.19820677179887136176716921909247e-01 4.24856830476827873610545793781057e-01 1.55322492343284990212737284309696e-01 6.57633915256547463279446219530655e-03
            3.41717950018184946081589714594884e-01 4.82050314988177130981483742289129e-01 1.76231734993637922936926543115987e-01 7.17048911587346484569627236282940e-03
            2.67714629312019669615807515583583e-01 5.36241864123632949734599151270231e-01 1.96043506564347436160744564404013e-01 7.44612351573773617596785356909095e-03
            1.99727347669159671639249609143008e-01 5.86027955863542726611115085688652e-01 2.14244696467297601749635305168340e-01 7.34697430479545107034899942277661e-03
            1.39516911332385140642742271666066e-01 6.30119177555702791870828605169663e-01 2.30363911111912122997580354422098e-01 6.84520810637235296164382347683386e-03
            8.86426717314284795534717886766884e-02 6.67373638960358395166849732049741e-01 2.43983689308213125279678479273571e-01 5.94528066296299510101741603307346e-03
            4.84220481925909940201790959690697e-02 6.96826613177797704601346140407259e-01 2.54751338629611301378474763623672e-01 4.68472033389438768935519874503370e-03
            1.98959239325848247226247167418478e-02 7.17715876655827700325573914597044e-01 2.62388199411587419440650137403281e-01 3.13182901900191986571675428763228e-03
            3.79657807820771298068507348943967e-03 7.29505192102634314466058640391566e-01 2.66698229819158028064407517376821e-01 1.38242224275846557757230215202071e-03
            9.96203421921792342530466157768387e-01 2.49921920023859307422031683643127e-03 1.29735887796906439531352539518139e-03 5.64380382436514189563097829149818e-06
            9.80104076067414897721619126969017e-01 1.30971295926243392476751026265447e-02 6.79879433996076303070577040443823e-03 6.81046633434529585272854124156083e-05
            9.51577951807409005979820904030930e-01 3.18753651485370395612939375951100e-02 1.65466830440539544588851583739597e-02 2.55369687226984169031718252540486e-04
            9.11357328268571409424225748807658e-01 5.83518796632299385773734456961392e-02 3.02907920681986519984008054962032e-02 6.19460186374408481972675399873651e-04
            8.60483088667614692823804034560453e-01 9.18414783990137290059507790829230e-02 4.76754329333715781702451863566239e-02 1.18893485691037251866797319621583e-03
            8.00272652330840550405355315888301e-01 1.31476927861084946780323434722959e-01 6.82504198080745028143212493887404e-02 1.96424189459628965870319206032946e-03
            7.32285370687980496917646178189898e-01 1.76231734993637922936926543115987e-01 9.14828943183815801454272786941146e-02 2.91614336999613133621922500537949e-03
            6.58282049981815053918410285405116e-01 2.24946792653554189511666550060909e-01 1.16771157364630756569923164533975e-01 3.98741511145484298084262775319075e-03
            5.80179322820112863823283078090753e-01 2.76360415998729902486985565701616e-01 1.43460261181157233689731356207631e-01 5.09768118256631770635056710716526e-03
            5.00000000000000111022302462515654e-01 3.29141024990907471448053911444731e-01 1.70858975009092417529643626039615e-01 6.15091271512329187953760367690847e-03
            4.19820677179887136176716921909247e-01 3.81921633983085151431424719703500e-01 1.98257688837027712391858358387253e-01 7.04483932597463509417412552693349e-03
            3.41717950018184946081589714594884e-01 4.33335257328260836651168119715294e-01 2.24946792653554217267242165689822e-01 7.68131669277918177618502681980317e-03
            2.67714629312019669615807515583583e-01 4.82050314988177075470332511031302e-01 2.50235055699803310425011204642942e-01 7.97658736156726076715894890867276e-03
            1.99727347669159671639249609143008e-01 5.26805122120729940604633156908676e-01 2.73467530210110387756117233948316e-01 7.87037473411889010321829118765891e-03
            1.39516911332385140642742271666066e-01 5.66440571582801477568125392281218e-01 2.94042517084813437300283567310544e-01 7.33286257650505191724166209610303e-03
            8.86426717314284795534717886766884e-02 5.99930170318585198607763686595717e-01 3.11427157949986321838764524727594e-01 6.36882405951633492890984200585081e-03
            4.84220481925909940201790959690697e-02 6.26406684833277993540434636088321e-01 3.25171266974131012439386267942609e-01 5.01846107291128214894504466769831e-03
            1.98959239325848247226247167418478e-02 6.45184920389190841305548929085489e-01 3.34919155678224278460675122914836e-01 3.35494136227543421693852998544116e-03
            3.79657807820771298068507348943967e-03 6.55782830781576508982766426925082e-01 3.40420591140215833547699730843306e-01 1.48090637586531146362156530216225e-03
            9.96203421921792342530466157768387e-01 2.20269609844823662306301237379103e-03 1.59388197975942084647082985782163e-03 5.87296073243102944097526751932037e-06
            9.80104076067414897721619126969017e-01 1.15432036740876695080082470212801e-02 8.35272025849743277037262600970280e-03 7.08699355893233014268631220211603e-05
            9.51577951807409005979820904030930e-01 2.80934711299403137851893319520968e-02 2.03285770626506802349897640169729e-02 2.65738532381744872214179853386895e-04
            9.11357328268571409424225748807658e-01 5.14286452581058303734629078007856e-02 3.72140264733227602023113433915569e-02 6.44612297503190621872359855615287e-04
            8.60483088667614692823804034560453e-01 8.09448271387770423856977686227765e-02 5.85720841936082647904981968167704e-02 1.23720950361675178195974833528226e-03
            8.00272652330840550405355315888301e-01 1.15877677319350141926790342949971e-01 8.38496703498093076678543411617284e-02 2.04399654469874939369256061638680e-03
            7.32285370687980496917646178189898e-01 1.55322492343285017968312899938610e-01 1.12392136968734485114040921871492e-01 3.03454833567906243074752126176463e-03
            6.58282049981815053918410285405116e-01 1.98257688837027712391858358387253e-01 1.43460261181157233689731356207631e-01 4.14931721623237274171325239535690e-03
            5.80179322820112863823283078090753e-01 2.43571276192108138181069421079883e-01 1.76249400987778997995647500829364e-01 5.30466372385511933384982796724216e-03
            5.00000000000000111022302462515654e-01 2.90089661410056376400490307787550e-01 2.09910338589943512577207229696796e-01 6.40065990397767434899600758058114e-03
            4.19820677179887136176716921909247e-01 3.36608046628004753397789272639784e-01 2.43571276192108110425493805450969e-01 7.33088286115065029119319817141331e-03
            3.41717950018184946081589714594884e-01 3.81921633983085151431424719703500e-01 2.76360415998729902486985565701616e-01 7.99320329230856357893841845907446e-03
            2.67714629312019669615807515583583e-01 4.24856830476827762588243331265403e-01 3.07428540211152623307100384408841e-01 8.30046291670312637311912595805552e-03
            1.99727347669159671639249609143008e-01 4.64301645500762583118614656996215e-01 3.35971006830077745242135733860778e-01 8.18993770893462931514950042810597e-03
            1.39516911332385140642742271666066e-01 4.99234495681335932459887771983631e-01 3.61248592986278982408521187608130e-01 7.63060079838468515583960183334966e-03
            8.86426717314284795534717886766884e-02 5.28750677562007109777653113269480e-01 3.82606650706564410668875098053832e-01 6.62741916220110395541187386925230e-03
            4.84220481925909940201790959690697e-02 5.52085851690172546568646794185042e-01 3.99492100117236459411174109845888e-01 5.22222701845187226715738759708074e-03
            1.98959239325848247226247167418478e-02 5.68636119146025298398683389677899e-01 4.11467956921389821367540662322426e-01 3.49116296267946209924692979598149e-03
            3.79657807820771298068507348943967e-03 5.77976626721664610286666174943093e-01 4.18226795200127732243799982825294e-01 1.54103602189646616237028720064473e-03
            9.96203421921792342530466157768387e-01 1.89828903910385692402340573892161e-03 1.89828903910380054551043649269104e-03 5.95001158435277681700093302397114e-06
            9.80104076067414897721619126969017e-01 9.94796196629252511833829686338504e-03 9.94796196629257716004257616759787e-03 7.17997202689045591365510357029223e-05
            9.51577951807409005979820904030930e-01 2.42110240962955039489834518917633e-02 2.42110240962954900711956440773065e-02 2.69224913653695380102298662094995e-04
            9.11357328268571409424225748807658e-01 4.43213358657143299823566451323131e-02 4.43213358657142605934176060600294e-02 6.53069348204651138958820588698018e-04
            8.60483088667614692823804034560453e-01 6.97584556661926674658857905342302e-02 6.97584556661926397103101749053167e-02 1.25344118821374612450514351991160e-03
            8.00272652330840550405355315888301e-01 9.98636738345797247973223420558497e-02 9.98636738345797247973223420558497e-02 2.07081294655624224138046329812823e-03
            7.32285370687980496917646178189898e-01 1.33857314656009779296752526533965e-01 1.33857314656009723785601295276138e-01 3.07436037344234035922907466442666e-03
            6.58282049981815053918410285405116e-01 1.70858975009092500796370472926355e-01 1.70858975009092445285219241668528e-01 4.20375456750543899375749390401324e-03
            5.80179322820112863823283078090753e-01 2.09910338589943623599509692212450e-01 2.09910338589943512577207229696796e-01 5.37425875057211815954971712017141e-03
            5.00000000000000111022302462515654e-01 2.50000000000000000000000000000000e-01 2.49999999999999888977697537484346e-01 6.48463395402358762364958266743997e-03
            4.19820677179887136176716921909247e-01 2.90089661410056487422792770303204e-01 2.90089661410056376400490307787550e-01 7.42706105738326523291181757713275e-03
            3.41717950018184946081589714594884e-01 3.29141024990907582470356373960385e-01 3.29141024990907471448053911444731e-01 8.09807086274116037782899724106755e-03
            2.67714629312019669615807515583583e-01 3.66142685343990248458823089094949e-01 3.66142685343990137436520626579295e-01 8.40936160821746832294554252484886e-03
            1.99727347669159671639249609143008e-01 4.00136326165420275202677657944150e-01 4.00136326165420053158072732912842e-01 8.29738635475560024068464315405436e-03
            1.39516911332385140642742271666066e-01 4.30241544333807568456506942311535e-01 4.30241544333807346411902017280227e-01 7.73071117183629467894956377449489e-03
            8.86426717314284795534717886766884e-02 4.55678664134285871245566568177310e-01 4.55678664134285649200961643146002e-01 6.71436820132378323761068728003920e-03
            4.84220481925909940201790959690697e-02 4.75788975903704614012212914531119e-01 4.75788975903704391967607989499811e-01 5.29074050918211351290931787616501e-03
            1.98959239325848247226247167418478e-02 4.90052038033707670905414488515817e-01 4.90052038033707448860809563484509e-01 3.53696559830525081649654772775193e-03
            3.79657807820771298068507348943967e-03 4.98101710960896282287535541399848e-01 4.98101710960896060242930616368540e-01 1.56125378662176736606348459446281e-03
            9.96203421921792342530466157768387e-01 1.59388197975947657446249561274954e-03 2.20269609844818067823091212176223e-03 5.87296073243105485196368514833054e-06
            9.80104076067414897721619126969017e-01 8.35272025849737725922139475187578e-03 1.15432036740877250191594782791071e-02 7.08699355893236131349877116036851e-05
            9.51577951807409005979820904030930e-01 2.03285770626506802349897640169729e-02 2.80934711299403137851893319520968e-02 2.65738532381746010626460963166551e-04
            9.11357328268571409424225748807658e-01 3.72140264733228157134625746493839e-02 5.14286452581057748623116765429586e-02 6.44612297503193440798008317926815e-04
            8.60483088667614692823804034560453e-01 5.85720841936082717293921007239987e-02 8.09448271387770423856977686227765e-02 1.23720950361675720297061076280443e-03
            8.00272652330840550405355315888301e-01 8.38496703498092521567031099039013e-02 1.15877677319350197437941574207798e-01 2.04399654469875850099080949462405e-03
            7.32285370687980496917646178189898e-01 1.12392136968734485114040921871492e-01 1.55322492343285017968312899938610e-01 3.03454833567907587485446008201961e-03
            6.58282049981815053918410285405116e-01 1.43460261181157233689731356207631e-01 1.98257688837027712391858358387253e-01 4.14931721623239008894801216342785e-03
            5.80179322820112863823283078090753e-01 1.76249400987778997995647500829364e-01 2.43571276192108138181069421079883e-01 5.30466372385514275261675365413794e-03
            5.00000000000000111022302462515654e-01 2.09910338589943512577207229696796e-01 2.90089661410056376400490307787550e-01 6.40065990397770210457162320949465e-03
            4.19820677179887136176716921909247e-01 2.43571276192108138181069421079883e-01 3.36608046628004697886638041381957e-01 7.33088286115068151621576575394101e-03
            3.41717950018184946081589714594884e-01 2.76360415998729902486985565701616e-01 3.81921633983085151431424719703500e-01 7.99320329230859827340793799521634e-03
            2.67714629312019669615807515583583e-01 3.07428540211152623307100384408841e-01 4.24856830476827762588243331265403e-01 8.30046291670316280231212147100450e-03
            1.99727347669159671639249609143008e-01 3.35971006830077745242135733860778e-01 4.64301645500762583118614656996215e-01 8.18993770893466400961901996424785e-03
            1.39516911332385140642742271666066e-01 3.61248592986278982408521187608130e-01 4.99234495681335932459887771983631e-01 7.63060079838471898294738338108800e-03
            8.86426717314284795534717886766884e-02 3.82606650706564410668875098053832e-01 5.28750677562007109777653113269480e-01 6.62741916220113257834922748656936e-03
            4.84220481925909940201790959690697e-02 3.99492100117236459411174109845888e-01 5.52085851690172546568646794185042e-01 5.22222701845189568592431328397652e-03
            1.98959239325848247226247167418478e-02 4.11467956921389821367540662322426e-01 5.68636119146025298398683389677899e-01 3.49116296267947727807734459304356e-03
            3.79657807820771298068507348943967e-03 4.18226795200127676732648751567467e-01 5.77976626721664610286666174943093e-01 1.54103602189647288442375661077222e-03
            9.96203421921792342530466157768387e-01 1.29735887796911990646475665300841e-03 2.49921920023853756306908557860424e-03 5.64380382436515121299339808880191e-06
            9.80104076067414897721619126969017e-01 6.79879433996070665219280115820766e-03 1.30971295926243964935498098611788e-02 6.81046633434530669475026609660517e-05
            9.51577951807409005979820904030930e-01 1.65466830440539509894382064203455e-02 3.18753651485370465001878415023384e-02 2.55369687226984602712587246742260e-04
            9.11357328268571409424225748807658e-01 3.02907920681987109789989887076445e-02 5.83518796632298830662222144383122e-02 6.19460186374409457754630636827642e-04
            8.60483088667614692823804034560453e-01 4.76754329333715851091390902638523e-02 9.18414783990137151281629712684662e-02 1.18893485691037447023188367012381e-03
            8.00272652330840550405355315888301e-01 6.82504198080744611809578259453701e-02 1.31476927861085002291474665980786e-01 1.96424189459629269446927501974187e-03
            7.32285370687980496917646178189898e-01 9.14828943183815940232150865085714e-02 1.76231734993637922936926543115987e-01 2.91614336999613610670878394159899e-03
            6.58282049981815053918410285405116e-01 1.16771157364630742692135356719518e-01 2.24946792653554217267242165689822e-01 3.98741511145484905237479367201558e-03
            5.80179322820112863823283078090753e-01 1.43460261181157233689731356207631e-01 2.76360415998729902486985565701616e-01 5.09768118256632637996794699120073e-03
            5.00000000000000111022302462515654e-01 1.70858975009092445285219241668528e-01 3.29141024990907471448053911444731e-01 6.15091271512330228787845953775104e-03
            4.19820677179887136176716921909247e-01 1.98257688837027712391858358387253e-01 3.81921633983085151431424719703500e-01 7.04483932597464636987671937617961e-03
            3.41717950018184946081589714594884e-01 2.24946792653554189511666550060909e-01 4.33335257328260836651168119715294e-01 7.68131669277919391924935865745283e-03
            2.67714629312019669615807515583583e-01 2.50235055699803310425011204642942e-01 4.82050314988177075470332511031302e-01 7.97658736156727291022328074632242e-03
            1.99727347669159671639249609143008e-01 2.73467530210110387756117233948316e-01 5.26805122120729940604633156908676e-01 7.87037473411890398100609900211566e-03
            1.39516911332385140642742271666066e-01 2.94042517084813437300283567310544e-01 5.66440571582801477568125392281218e-01 7.33286257650506406030599393375269e-03
            8.86426717314284795534717886766884e-02 3.11427157949986266327613293469767e-01 5.99930170318585309630066149111371e-01 6.36882405951634533725069786669337e-03
            4.84220481925909940201790959690697e-02 3.25171266974131012439386267942609e-01 6.26406684833277993540434636088321e-01 5.01846107291128995520068656333024e-03
            1.98959239325848247226247167418478e-02 3.34919155678224278460675122914836e-01 6.45184920389190841305548929085489e-01 3.35494136227543942110895791586245e-03
            3.79657807820771298068507348943967e-03 3.40420591140215833547699730843306e-01 6.55782830781576508982766426925082e-01 1.48090637586531384886634477027201e-03
            9.96203421921792342530466157768387e-01 1.01639949286151795235233485925619e-03 2.78017858534613951718150737235646e-03 5.26847616278835407030564294550423e-06
            9.80104076067414897721619126969017e-01 5.32642990043214609174881957187608e-03 1.45694940321529561866320534591068e-02 6.35755257563491016489440332826177e-05
            9.51577951807409005979820904030930e-01 1.29632906824082502761363855370291e-02 3.54587575101827437440427104320406e-02 2.38386937555439497706324947223777e-04
            9.11357328268571409424225748807658e-01 2.37309400038064645388491413768861e-02 6.49117317276221295063720617690706e-02 5.78264469721733831257537428172100e-04
            8.60483088667614692823804034560453e-01 3.73507182001074494737125064602878e-02 1.02166193132277857702483458979259e-01 1.10986759066613888818908506550542e-03
            8.00272652330840550405355315888301e-01 5.34699328447218805160368049200770e-02 1.46257414824437576017501783098851e-01 1.83361468996397496307448893304581e-03
            7.32285370687980496917646178189898e-01 7.16711227476720530438214495916327e-02 1.96043506564347436160744564404013e-01 2.72221223667818302410781328148914e-03
            6.58282049981815053918410285405116e-01 9.14828943183816495343663177663984e-02 2.50235055699803310425011204642942e-01 3.72224161568992899240537752802993e-03
            5.80179322820112863823283078090753e-01 1.12392136968734554502979960943776e-01 3.07428540211152567795949153151014e-01 4.75867209981673585511563473460228e-03
            5.00000000000000111022302462515654e-01 1.33857314656009807052328142162878e-01 3.66142685343990081925369395321468e-01 5.74186099082990824421512954245372e-03
            4.19820677179887136176716921909247e-01 1.55322492343285101235039746825350e-01 4.24856830476827762588243331265403e-01 6.57633915256549198002922196337749e-03
            3.41717950018184946081589714594884e-01 1.76231734993638033959229005631641e-01 4.82050314988177019959181279773475e-01 7.17048911587348306029277011930390e-03
            2.67714629312019669615807515583583e-01 1.96043506564347574938622642548580e-01 5.36241864123632838712296688754577e-01 7.44612351573775525792608931396899e-03
            1.99727347669159671639249609143008e-01 2.14244696467297712771937767683994e-01 5.86027955863542615588812623172998e-01 7.34697430479546928494549717925111e-03
            1.39516911332385140642742271666066e-01 2.30363911111912289531034048195579e-01 6.30119177555702680848526142654009e-01 6.84520810637237030887858324490480e-03
            8.86426717314284795534717886766884e-02 2.43983689308213236301980941789225e-01 6.67373638960358284144547269534087e-01 5.94528066296301071352869982433731e-03
            4.84220481925909940201790959690697e-02 2.54751338629611412400777226139326e-01 6.96826613177797593579043677891605e-01 4.68472033389439896505779259427982e-03
            1.98959239325848247226247167418478e-02 2.62388199411587585974103831176762e-01 7.17715876655827589303271452081390e-01 3.13182901900192767197239618326421e-03
            3.79657807820771298068507348943967e-03 2.66698229819158139086709979892476e-01 7.29505192102634203443756177875912e-01 1.38242224275846904701925410563490e-03
            9.96203421921792342530466157768387e-01 7.58280469779302004375287626913860e-04 3.03829760842835557357877185324924e-03 4.75669917446966059715708977084780e-06
            9.80104076067414897721619126969017e-01 3.97376011648256817487823155943261e-03 1.59221638161025358382261174483574e-02 5.73998328050988868877235749010879e-05
            9.51577951807409005979820904030930e-01 9.67120725421442666014382183448106e-03 3.87508409383765656253117981577816e-02 2.15230156507754168951004891674472e-04
            9.11357328268571409424225748807658e-01 1.77043657152262425003019785663128e-02 7.09383060162023515449192245796439e-02 5.22092164937257578911922184516925e-04
            8.60483088667614692823804034560453e-01 2.78653426554106437018631936552993e-02 1.11651568676974663474332771784248e-01 1.00205563984143912296953526919197e-03
            8.00272652330840550405355315888301e-01 3.98910134069573249493423361400346e-02 1.59836334262202117706408444064436e-01 1.65549832865352999390629040732392e-03
            7.32285370687980496917646178189898e-01 5.34699328447219221494002283634472e-02 2.14244696467297573994059689539426e-01 2.45777797959802635913151291902068e-03
            6.58282049981815053918410285405116e-01 6.82504198080745583254724806465674e-02 2.73467530210110387756117233948316e-01 3.36066503357930696296862471683653e-03
            5.80179322820112863823283078090753e-01 8.38496703498093631790055724195554e-02 3.35971006830077745242135733860778e-01 4.29641720857481135970035168725190e-03
            5.00000000000000111022302462515654e-01 9.98636738345798080640491889425903e-02 4.00136326165420053158072732912842e-01 5.18409965065592427679863618550371e-03
            4.19820677179887136176716921909247e-01 1.15877677319350308460244036723452e-01 4.64301645500762583118614656996215e-01 5.93751704505860074256862191077744e-03
            3.41717950018184946081589714594884e-01 1.31476927861085113313777128496440e-01 5.26805122120729940604633156908676e-01 6.47395159513589144723022528182810e-03
            2.67714629312019669615807515583583e-01 1.46257414824437714795379861243418e-01 5.86027955863542615588812623172998e-01 6.72281101516141853940977668457890e-03
            1.99727347669159671639249609143008e-01 1.59836334262202284239862137837918e-01 6.40436318068638099632039484276902e-01 6.63329310613687030129259625255145e-03
            1.39516911332385140642742271666066e-01 1.71862005013749069570749838931079e-01 6.88621083653865873053234736289596e-01 6.18026821087897058198956656838163e-03
            8.86426717314284795534717886766884e-02 1.82022981953933460363970198159222e-01 7.29334346314638004571406781906262e-01 5.36775924341268643319446951522877e-03
            4.84220481925909940201790959690697e-02 1.90056140414945257122170119146176e-01 7.61521811392463776613226400513668e-01 4.22964907808616973061699439995209e-03
            1.98959239325848247226247167418478e-02 1.95753587552677132954670469189296e-01 7.84350488514737986811553582811030e-01 2.82760480430497684184443230037687e-03
            3.79657807820771298068507348943967e-03 1.98969067199380372779060621724057e-01 7.97234354722411997506981151673244e-01 1.24813447716492381157904478783394e-03
            9.96203421921792342530466157768387e-01 5.29686847103782651458026009549940e-04 3.26689123110387481807581622206271e-03 4.12172922919725862641714825285177e-06
            9.80104076067414897721619126969017e-01 2.77581785517834727511354842022229e-03 1.71201060774067541359055866223571e-02 4.97375511770070048277875285336336e-05
            9.51577951807409005979820904030930e-01 6.75569460421819763251694013206361e-03 4.16663535883727981223856318138132e-02 1.86499165607128966843167305889040e-04
            9.11357328268571409424225748807658e-01 1.23671517722194531513357773633288e-02 7.62755199592091426286089017594350e-02 4.52398282428002223611029908312275e-04
            8.60483088667614692823804034560453e-01 1.94649685477286393808782349879039e-02 1.20051942784656667795317730451643e-01 8.68291617469567973859601028863153e-04
            8.00272652330840550405355315888301e-01 2.78653426554105708434772026294013e-02 1.71862005013748875281720529528684e-01 1.43450649280532616623717867554433e-03
            7.32285370687980496917646178189898e-01 3.73507182001073800847734673880041e-02 2.30363911111912122997580354422098e-01 2.12969014138171238229024773147557e-03
            6.58282049981815053918410285405116e-01 4.76754329333715295979878590060252e-02 2.94042517084813437300283567310544e-01 2.91205114941694639951741407912778e-03
            5.80179322820112863823283078090753e-01 5.85720841936082023404530616517150e-02 3.61248592986278926897369956350303e-01 3.72289012608902026776891425186022e-03
            5.00000000000000111022302462515654e-01 6.97584556661925564435833280185761e-02 4.30241544333807346411902017280227e-01 4.49207618002501595649844645663507e-03
            4.19820677179887136176716921909247e-01 8.09448271387769452411831139215792e-02 4.99234495681335932459887771983631e-01 5.14492017591243206570128521093466e-03
            3.41717950018184946081589714594884e-01 9.18414783990136179836483165672689e-02 5.66440571582801477568125392281218e-01 5.60974628399848063847743517840172e-03
            2.67714629312019669615807515583583e-01 1.02166193132277746680180996463605e-01 6.30119177555702680848526142654009e-01 5.82538555565678423803221974708322e-03
            1.99727347669159671639249609143008e-01 1.11651568676974524696454693639680e-01 6.88621083653865762030932273773942e-01 5.74781735791507226301932931278316e-03
            1.39516911332385140642742271666066e-01 1.20051942784656529017439652307075e-01 7.40431145882958441362120538542513e-01 5.35526658187262909194270577017960e-03
            8.86426717314284795534717886766884e-02 1.27149759560165720451152537862072e-01 7.84207568708405799995375673461240e-01 4.65121912430683723260749218297860e-03
            4.84220481925909940201790959690697e-02 1.32761216728166930867160999696353e-01 8.18816735079242130623811135592405e-01 3.66503484768693904571890129773237e-03
            1.98959239325848247226247167418478e-02 1.36741093477206809847501745025511e-01 8.43362982590208254407571075716987e-01 2.45014892534628839215748996593902e-03
            3.79657807820771298068507348943967e-03 1.38987224485281363195454673586937e-01 8.57216197436511007090587099810364e-01 1.08152148534242293666240986027560e-03
            9.96203421921792342530466157768387e-01 3.36538824289303909690490579365019e-04 3.46003925391835361405346027652286e-03 3.38001644175229110886619562914568e-06
            9.80104076067414897721619126969017e-01 1.76362785394960789112084320606755e-03 1.81322960786354930862174228423100e-02 4.07871869796554468393795511715894e-05
            9.51577951807409005979820904030930e-01 4.29225972249925279144999734626253e-03 4.41297884700917394940056226460001e-02 1.52938296300446016640073310099979e-04
            9.11357328268571409424225748807658e-01 7.85752325168580408043528251482712e-03 8.07851484797427882300624446543225e-02 3.70988375945537768307858073058014e-04
            8.60483088667614692823804034560453e-01 1.23671517722194444777183974792933e-02 1.27149759560165859229030616006639e-01 7.12040937209845585000034251521583e-04
            8.00272652330840550405355315888301e-01 1.77043657152261835197037953548715e-02 1.82022981953933266074940888756828e-01 1.17636439995519401918722124378291e-03
            7.32285370687980496917646178189898e-01 2.37309400038064055582509581654449e-02 2.43983689308213097524102863644657e-01 1.74644846699692035500717235407819e-03
            6.58282049981815053918410285405116e-01 3.02907920681986519984008054962032e-02 3.11427157949986321838764524727594e-01 2.38802216665016118005149259317932e-03
            5.80179322820112863823283078090753e-01 3.72140264733227393856296316698717e-02 3.82606650706564410668875098053832e-01 3.05294917188649907180408860085663e-03
            5.00000000000000111022302462515654e-01 4.43213358657142328378419904311158e-02 4.55678664134285649200961643146002e-01 3.68371877476421534514106070901107e-03
            4.19820677179887136176716921909247e-01 5.14286452581057401678421570068167e-02 5.28750677562007109777653113269480e-01 4.21908228781781111799142891527481e-03
            3.41717950018184946081589714594884e-01 5.83518796632298275550709831804852e-02 5.99930170318585198607763686595717e-01 4.60026207924058710974080099731509e-03
            2.67714629312019669615807515583583e-01 6.49117317276220601174330226967868e-02 6.67373638960358284144547269534087e-01 4.77709666568781627510320575424885e-03
            1.99727347669159671639249609143008e-01 7.09383060162022682781923776929034e-02 7.29334346314638004571406781906262e-01 4.71348700839474381296678728858751e-03
            1.39516911332385140642742271666066e-01 7.62755199592090593618820548726944e-02 7.84207568708405799995375673461240e-01 4.39157646952499647330370180497994e-03
            8.86426717314284795534717886766884e-02 8.07851484797426910855477899531252e-02 8.30572179788828801605404805741273e-01 3.81422365229256918328637482318300e-03
            4.84220481925909940201790959690697e-02 8.43504120089292319661922192608472e-02 8.67227539798479774013628684770083e-01 3.00550505768900364303441996582933e-03
            1.98959239325848247226247167418478e-02 8.68790438774788842390961463024723e-02 8.93225032189936207771552290068939e-01 2.00924010091392199472970503393299e-03
            3.79657807820771298068507348943967e-03 8.83061329071391787337574896810111e-02 9.07897289014653163796708668087376e-01 8.86899696532879788443692881116931e-04
            9.96203421921792342530466157768387e-01 1.83838086669908389340563625857783e-04 3.61273999153774926992865879071815e-03 2.55078844149308508511145646147078e-06
            9.80104076067414897721619126969017e-01 9.63401387499757664613364571692955e-04 1.89325225450853451558685947020422e-02 3.07807630233850352749480627778667e-05
            9.51577951807409005979820904030930e-01 2.34469475116560457580994203397040e-03 4.60773534414253907454117609177047e-02 1.15417556449097407950295235412597e-04
            9.11357328268571409424225748807658e-01 4.29225972249926146506737723029801e-03 8.43504120089293291107068739620445e-02 2.79972857409999792682037123014993e-04
            8.60483088667614692823804034560453e-01 6.75569460421820630613432001609908e-03 1.32761216728167097400614693469834e-01 5.37354129426416936288202208515941e-04
            8.00272652330840550405355315888301e-01 9.67120725421441278235601402002430e-03 1.90056140414945035077565194114868e-01 8.87763940235165513459314468036609e-04
            7.32285370687980496917646178189898e-01 1.29632906824082416025190056529937e-02 2.54751338629611245867323532365845e-01 1.31798783823949806036368581629858e-03
            6.58282049981815053918410285405116e-01 1.65466830440539509894382064203455e-02 3.25171266974131012439386267942609e-01 1.80216263609729414330029584334625e-03
            5.80179322820112863823283078090753e-01 2.03285770626506802349897640169729e-02 3.99492100117236459411174109845888e-01 2.30396141389086189094070888927490e-03
            5.00000000000000111022302462515654e-01 2.42110240962954900711956440773065e-02 4.75788975903704391967607989499811e-01 2.77998271141790403992999358706584e-03
            4.19820677179887136176716921909247e-01 2.80934711299403137851893319520968e-02 5.52085851690172546568646794185042e-01 3.18400413694276945147931812130082e-03
            3.41717950018184946081589714594884e-01 3.18753651485370395612939375951100e-02 6.26406684833277993540434636088321e-01 3.47166812404097781252776400151561e-03
            2.67714629312019669615807515583583e-01 3.54587575101827437440427104320406e-02 6.96826613177797593579043677891605e-01 3.60511943321033690584842545945321e-03
            1.99727347669159671639249609143008e-01 3.87508409383765656253117981577816e-02 7.61521811392463776613226400513668e-01 3.55711529435875483534656993356293e-03
            1.39516911332385140642742271666066e-01 4.16663535883727981223856318138132e-02 8.18816735079242130623811135592405e-01 3.31417988386765441019554678803161e-03
            8.86426717314284795534717886766884e-02 4.41297884700917394940056226460001e-02 8.67227539798479774013628684770083e-01 2.87847049657945089698452001414353e-03
            4.84220481925909940201790959690697e-02 4.60773534414253907454117609177047e-02 9.05500598365983622173303047020454e-01 2.26815688447587785955827932582451e-03
            1.98959239325848247226247167418478e-02 4.74586468050912438365607215473574e-02 9.32645429262323855112981618731283e-01 1.51630813456594287259859843430831e-03
            3.79657807820771298068507348943967e-03 4.82382101059210879889782574991841e-02 9.47965211815871233724806188547518e-01 6.69314346147665824131312017897244e-04
            9.96203421921792342530466157768387e-01 7.55364286481397320671682393822266e-05 3.72104164955951773746667399223043e-03 1.65562124269201559833522997139399e-06
            9.80104076067414897721619126969017e-01 3.95847789131206025028258954634452e-04 1.95000761434538961991425054520732e-02 1.99786404465418393505173055846313e-05
            9.51577951807409005979820904030930e-01 9.63401387499746931011856965199058e-04 4.74586468050912438365607215473574e-02 7.49132131572929292053983352239754e-05
            9.11357328268571409424225748807658e-01 1.76362785394959141124782142640015e-03 8.68790438774789952613986088181264e-02 1.81719895921221541966461554551415e-04
            8.60483088667614692823804034560453e-01 2.77581785517831995321880178551055e-03 1.36741093477206976380955438798992e-01 3.48776439886130907534061984520690e-04
            8.00272652330840550405355315888301e-01 3.97376011648251786789742823202687e-03 1.95753587552676938665641159786901e-01 5.76214324183219137737177462099680e-04
            7.32285370687980496917646178189898e-01 5.32642990043208364170368440682068e-03 2.62388199411587419440650137403281e-01 8.55456543201901743397885802266956e-04
            6.58282049981815053918410285405116e-01 6.79879433996063119172159616709905e-03 3.34919155678224333971826354172663e-01 1.16971627069238051817301471402288e-03
            5.80179322820112863823283078090753e-01 8.35272025849728358415369200429268e-03 4.11467956921389876878691893580253e-01 1.49541506348823652512713966444835e-03
            5.00000000000000111022302462515654e-01 9.94796196629241062658888239411681e-03 4.90052038033707504371960794742336e-01 1.80438265928707609539460854364279e-03
            4.19820677179887136176716921909247e-01 1.15432036740875411384710247375551e-02 5.68636119146025298398683389677899e-01 2.06661783478055520457683513768643e-03
            3.41717950018184946081589714594884e-01 1.30971295926241935309031205747488e-02 6.45184920389190841305548929085489e-01 2.25332975492650768728486454506310e-03
            2.67714629312019669615807515583583e-01 1.45694940321527393461975563582200e-02 7.17715876655827700325573914597044e-01 2.33994800155637016486620538557872e-03
            1.99727347669159671639249609143008e-01 1.59221638161022999158333846025926e-02 7.84350488514737986811553582811030e-01 2.30879031292685176629531085268354e-03
            1.39516911332385140642742271666066e-01 1.71201060774065078051719979157497e-02 8.43362982590208365429873538232641e-01 2.15111003663716457076637844636480e-03
            8.86426717314284795534717886766884e-02 1.81322960786352363471429782748601e-02 8.93225032189936318793854752584593e-01 1.86830739197235500513583073001200e-03
            4.84220481925909940201790959690697e-02 1.89325225450850780084532942737496e-02 9.32645429262323966135284081246937e-01 1.47217568443203366407301402318808e-03
            1.98959239325848247226247167418478e-02 1.95000761434536221128333011165523e-02 9.60603999923961504592284654791001e-01 9.84178819857210599245100901555361e-04
            3.79657807820771298068507348943967e-03 1.98203875039366858579192864908691e-02 9.76383034417855677489228582999203e-01 4.34426874253811506725042601573250e-04
            9.96203421921792342530466157768387e-01 1.44140051039273714401185363942481e-05 3.78216407310373028613503798567308e-03 7.18998236483508931737808258372047e-07
            9.80104076067414897721619126969017e-01 7.55364286481405858763790717169684e-05 1.98203875039369634136754427800042e-02 8.67626415873056088695104703933936e-06
            9.51577951807409005979820904030930e-01 1.83838086669908389340563625857783e-04 4.82382101059210879889782574991841e-02 3.25330859259979267186606299855356e-05
            9.11357328268571409424225748807658e-01 3.36538824289304560211794070667679e-04 8.83061329071392897560599521966651e-02 7.89167723463609944046376343251836e-05
            8.60483088667614692823804034560453e-01 5.29686847103783301979329500852600e-04 1.38987224485281529728908367360418e-01 1.51465588105995178089605013305174e-04
            8.00272652330840550405355315888301e-01 7.58280469779300920173115141409426e-04 1.98969067199380150734455696692748e-01 2.50236631568359548394991920261532e-04
            7.32285370687980496917646178189898e-01 1.01639949286151730183103136795353e-03 2.66698229819157972553256286118994e-01 3.71505106415733377285548488799805e-04
            6.58282049981815053918410285405116e-01 1.29735887796911990646475665300841e-03 3.40420591140215833547699730843306e-01 5.07980880002720865291909735361742e-04
            5.80179322820112863823283078090753e-01 1.59388197975947657446249561274954e-03 4.18226795200127676732648751567467e-01 6.49424376622914070397718511173935e-04
            5.00000000000000111022302462515654e-01 1.89828903910385605666166775051806e-03 4.98101710960896060242930616368540e-01 7.83601899103060104898044180288252e-04
            4.19820677179887136176716921909247e-01 2.20269609844823662306301237379103e-03 5.77976626721664610286666174943093e-01 8.97484606005987404012125274732625e-04
            3.41717950018184946081589714594884e-01 2.49921920023859307422031683643127e-03 6.55782830781576508982766426925082e-01 9.78569299686957638426121341979069e-04
            2.67714629312019669615807515583583e-01 2.78017858534619546201360762438526e-03 7.29505192102634203443756177875912e-01 1.01618561250552304209282539204651e-03
            1.99727347669159671639249609143008e-01 3.03829760842841108473000311107626e-03 7.97234354722411886484678689157590e-01 1.00265454477103314705199732514984e-03
            1.39516911332385140642742271666066e-01 3.26689123110393032922704747988973e-03 8.57216197436511007090587099810364e-01 9.34177626465626130200514687373925e-04
            8.86426717314284795534717886766884e-02 3.46003925391840912520469153434988e-03 9.07897289014653163796708668087376e-01 8.11362940628271604379118109307001e-04
            4.84220481925909940201790959690697e-02 3.61273999153780478107989004854517e-03 9.47965211815871233724806188547518e-01 6.39332048663161117176001457806933e-04
            1.98959239325848247226247167418478e-02 3.72104164955957324861790525005745e-03 9.76383034417855566466926120483549e-01 4.27406231337773394058504594639203e-04
            3.79657807820771298068507348943967e-03 3.78216407310378579728626924350010e-03 9.92421257848688509461965168156894e-01 1.88661602312889872942389124510498e-04
            ];
        
    case  37
        
        % ALG. DEG.:   37
        % PTS CARD.:  400
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.5e-15
        
        xyw_bar=[
            9.96564299592547442330214835237712e-01 3.42389637016278806577740390082454e-03 1.18040372897696040077608614637938e-05 5.32968604421356704037933318257592e-07
            9.81985963638956738108731769898441e-01 1.79521455289777601138645479750267e-02 6.18908320655017774036821265326580e-05 6.44140394826530023785404396874377e-06
            9.56117214125662862045373913133517e-01 4.37320177690285263882863375783927e-02 1.50768105308611566339749288090388e-04 2.42212332663714091684267065707203e-05
            9.19558485911109335653179641667521e-01 8.01651411461593577101325536204968e-02 2.76372942731306636687804711982608e-04 5.89972994537354606162625481058370e-05
            8.73165953230075397861753572215093e-01 1.26398282983758325492829044378595e-01 4.35763786166276645417383406311274e-04 1.13858787955958639217038386171055e-04
            8.18026840363257568000676656083670e-01 1.81347954378032999445125028614711e-01 6.25205258709432554198315301619004e-04 1.89423079811148147412816489243426e-04
            7.55433500975413618760967437992804e-01 2.43726241804238352717248972112429e-01 8.40257220348028521783589894766919e-04 2.83643898916982714793200148761798e-04
            6.86853044357709663358946272637695e-01 3.12071076519197576626396539722919e-01 1.07587912309276001465718763938639e-03 3.91884921150247120952281321137889e-04
            6.13892925570822534098169853677973e-01 3.84780526196240812986104629089823e-01 1.32654823293665291572551723220386e-03 5.07254742252743811778414873003840e-04
            5.38263260566748669155856532597682e-01 4.60150350329444679520207728273817e-01 1.58638910380665132393573912850115e-03 6.21174262743101526090816477676526e-04
            4.61736739433251219821841004886664e-01 5.36413949263102818321158338221721e-01 1.84931130364596185700065689161420e-03 7.24125362982039224961361423282824e-04
            3.86107074429177521412981377579854e-01 6.11783773396306518321807743632235e-01 2.10915217451601577636211004573852e-03 8.06512282095799525050150435845353e-04
            3.13146955642290225618751264846651e-01 6.84493223073349921214969526772620e-01 2.35982128435985316627920838072896e-03 8.59556020839622922889711187366402e-04
            2.44566499024586769817091180811985e-01 7.52838057788308590012604781804839e-01 2.59544318710458465915280612534843e-03 8.76138409977530595480754715964622e-04
            1.81973159636742376488172112658503e-01 8.15216345214514359618362959736260e-01 2.81049514874320838231369634740986e-03 8.51516585078317686866022917513419e-04
            1.26834046769924602138246427784907e-01 8.70166016608789116837385790859116e-01 2.99993662128628102436778135597706e-03 7.83840141121803319831928824612532e-04
            8.04415140888906643468203583324794e-02 9.16399158446388084620082281617215e-01 3.15932746472125103309736005030572e-03 6.74421260874965102817546469538001e-04
            4.38827858743368603988699305773480e-02 9.52832281823519089414276095340028e-01 3.28493230214399467570274282479659e-03 5.27731720124768226634626167026454e-04
            1.80140363610431508689657675859053e-02 9.78612154063569827933122269314481e-01 3.37380957538702119791196309961379e-03 3.51135533233636911148306536389896e-04
            3.43570040745255766978516476228833e-03 9.93140403222384615666840090852929e-01 3.42389637016282666337474438478239e-03 1.54593655144627366033321935745448e-04
            9.96564299592547442330214835237712e-01 3.37380957538705632606235162995745e-03 6.18908320655013437228131323308844e-05 1.22852722877150498452221774603466e-06
            9.81985963638956738108731769898441e-01 1.76895308550262739255565236362600e-02 3.24505506016987965711706465299358e-04 1.48478542193902998832209794155546e-05
            9.56117214125662862045373913133517e-01 4.30922797739729523947005418449407e-02 7.90506100364185559925545021542348e-04 5.58315149059670948021583125608913e-05
            9.19558485911109335653179641667521e-01 7.89924377291560120273317124883761e-02 1.44907635973465231948864584410330e-03 1.35992604820675078245578726310328e-04
            8.73165953230075397861753572215093e-01 1.24549253639592918951883859790541e-01 2.28479313033168318636256799436524e-03 2.62451896938061337866826550069277e-04
            8.18026840363257568000676656083670e-01 1.78695088522312167667038806939672e-01 3.27807111443026433228453697665827e-03 4.36632494625847444955774045993735e-04
            7.55433500975413618760967437992804e-01 2.40160869218464440022486883208330e-01 4.40562980612194121654567879886599e-03 6.53817598642143023475459262527920e-04
            6.86853044357709663358946272637695e-01 3.07505914997000140420624347825651e-01 5.64104064529019622042937953665387e-03 9.03320180934021554029766942761626e-04
            6.13892925570822534098169853677973e-01 3.79151727551154227313645606045611e-01 6.95534687802323858818454027641565e-03 1.16925510735768210975049630206968e-03
            5.38263260566748669155856532597682e-01 4.53418997019871206077823444502428e-01 8.31774241338012476632002289989032e-03 1.43184699672974915726775257951431e-03
            4.61736739433251219821841004886664e-01 5.28566966619085643053210787911667e-01 9.69629394766313712494820720166899e-03 1.66915596545003010880980909291793e-03
            3.86107074429177521412981377579854e-01 6.02834236087802399772783701337175e-01 1.10586894830201343253861523407977e-02 1.85906316183308663661510884423933e-03
            3.13146955642290225618751264846651e-01 6.74480048641956764221561115846271e-01 1.23729957157530101596876193070784e-02 1.98133242276521837760272504169734e-03
            2.44566499024586769817091180811985e-01 7.41825094420491826241459420998581e-01 1.36084065549213484302981669316068e-02 2.01955590610927411890362925817044e-03
            1.81973159636742376488172112658503e-01 8.03290875116644542686117347329855e-01 1.47359652466130253145593087538145e-02 1.96280100148676266494374154092384e-03
            1.26834046769924602138246427784907e-01 8.57436709999363833034635717922356e-01 1.57292432307115648271178542927373e-02 1.80680240521433467353051849357826e-03
            8.04415140888906643468203583324794e-02 9.02993525909800753836975673038978e-01 1.65649600013085818162039686285425e-02 1.55458478374510497034555189799221e-03
            4.38827858743368603988699305773480e-02 9.38893683864983952247484921826981e-01 1.72235302606791318424939163378440e-02 1.21645587053592976745086673417973e-03
            1.80140363610431508689657675859053e-02 9.64296432783930623777735036128433e-01 1.76895308550262253532991962856613e-02 8.09390196698495708454990094082859e-04
            3.43570040745255766978516476228833e-03 9.78612154063569716910819806798827e-01 1.79521455289777254193950284388848e-02 3.56348409953125466655826647510708e-04
            9.96564299592547442330214835237712e-01 3.28493230214394436872193949739085e-03 1.50768105308613301063225264897483e-04 1.89634498686058429013309451632852e-06
            9.81985963638956738108731769898441e-01 1.72235302606790763313426850800170e-02 7.90506100364185559925545021542348e-04 2.29190312230470312308827640634945e-05
            9.56117214125662862045373913133517e-01 4.19570869782442124051868859169190e-02 1.92569889609292554943920094956411e-03 8.61810881527108867822262183722160e-05
            9.19558485911109335653179641667521e-01 7.69115163507203963844105487623892e-02 3.52999773817026796240980957009015e-03 2.09917117311011358236558232981395e-04
            8.73165953230075397861753572215093e-01 1.21268215453944339099479066135245e-01 5.56583131598026303876736164966132e-03 4.05118687966101993867124031467597e-04
            8.18026840363257568000676656083670e-01 1.73987670437526636968428306317946e-01 7.98548919921579503089503759838408e-03 6.73982491305578088405803249827386e-04
            7.55433500975413618760967437992804e-01 2.33834239715854180774456949620799e-01 1.07322593087322004645756123863976e-02 1.00922771304473601373941082925967e-03
            6.86853044357709663358946272637695e-01 2.99405194840639155007266936081578e-01 1.37417608016511816337867912807269e-02 1.39435794057018135061465358148780e-03
            6.13892925570822534098169853677973e-01 3.69163620357435107166566012892872e-01 1.69434540717423587352641334291548e-02 1.80485300550979848970001295782595e-03
            5.38263260566748669155856532597682e-01 4.41474444966387380429040376839112e-01 2.02622944668639504151030905632069e-02 2.21018778469814431222872919136080e-03
            4.61736739433251219821841004886664e-01 5.14642769159275648149787230067886e-01 2.36204914074731320283717650454491e-02 2.57649604602969671540702023548874e-03
            3.86107074429177521412981377579854e-01 5.86953593768227643856505437724991e-01 2.69393318025948902416644159529824e-02 2.86963530366737872359661487564608e-03
            3.13146955642290225618751264846651e-01 6.56712019285023762549258208309766e-01 3.01410250726860118319905268435832e-02 3.05836917507506036995046372339857e-03
            2.44566499024586769817091180811985e-01 7.22282974409808264937282729079016e-01 3.31505265656049097344748588511720e-02 3.11737064392515013031004222909814e-03
            1.81973159636742376488172112658503e-01 7.82129543688136141810218759928830e-01 3.58972966751214261904578961548395e-02 3.02976421865423914056569820729692e-03
            1.26834046769924602138246427784907e-01 8.34848998671718578457046078256099e-01 3.83169545583568194047074939589947e-02 2.78896600997771953317494464386073e-03
            8.04415140888906643468203583324794e-02 8.79205697774942507294326787814498e-01 4.03527881361668283588528538530227e-02 2.39964486929013925964326858775166e-03
            4.38827858743368603988699305773480e-02 9.14160127147418899440367567876820e-01 4.19570869782441846496112702880055e-02 1.87771173304371753048913618044935e-03
            1.80140363610431508689657675859053e-02 9.38893683864983841225182459311327e-01 4.30922797739730079058517731027678e-02 1.24936835421884571105388683776027e-03
            3.43570040745255766978516476228833e-03 9.52832281823518867369671170308720e-01 4.37320177690285749605436649289913e-02 5.50056608404269177861611872515368e-04
            9.96564299592547442330214835237712e-01 3.15932746472125536990604999232346e-03 2.76372942731302299879114769964872e-04 2.51980644655460147413299915042373e-06
            9.81985963638956738108731769898441e-01 1.65649600013086130412265362110702e-02 1.44907635973464885004169389048911e-03 3.04541225487817237324150027033554e-05
            9.56117214125662862045373913133517e-01 4.03527881361668769311101812036213e-02 3.52999773817026102351590566286177e-03 1.14514849883828889352474500462620e-04
            9.19558485911109335653179641667521e-01 7.39706768999774666450974791587214e-02 6.47083718891319770172287917375797e-03 2.78931581071715812159689251714667e-04
            8.73165953230075397861753572215093e-01 1.16631324009730696911191216713632e-01 1.02027227601939052270552110712742e-02 5.38309584294945921677244538017248e-04
            8.18026840363257568000676656083670e-01 1.67334963152023424859393685437681e-01 1.46381964847190071399296584786498e-02 8.95567757039970749680102279910443e-04
            7.55433500975413618760967437992804e-01 2.24893199547629457635267158366332e-01 1.96732994769569236037654036408640e-02 1.34103157125525936609056198989265e-03
            6.86853044357709663358946272637695e-01 2.87956940398097827049639363394817e-01 2.51900152441925095914143639674876e-02 1.85278108772285714928029776160656e-03
            6.13892925570822534098169853677973e-01 3.55048036761662411109341519477312e-01 3.10590376675150547924886268447153e-02 2.39823464078447708483499134501926e-03
            5.38263260566748669155856532597682e-01 4.24593937002772980893183785155998e-01 3.71428024304783499509596822463209e-02 2.93683136062629183971051105572769e-03
            4.61736739433251219821841004886664e-01 4.94964548908336410271147087769350e-01 4.32987116584123699070119073439855e-02 3.42357081189962296485784776223227e-03
            3.86107074429177521412981377579854e-01 5.64510449149446813521535659674555e-01 4.93824764213757205766341940034181e-02 3.81308548156767193604288479491515e-03
            3.13146955642290225618751264846651e-01 6.31601545513011619625842740788357e-01 5.52514988446981547554059943649918e-02 4.06386939965822872183887071173558e-03
            2.44566499024586769817091180811985e-01 6.94665286363479461684278248867486e-01 6.07682146119337129874793390627019e-02 4.14226878510484848583406147781716e-03
            1.81973159636742376488172112658503e-01 7.52223522759085883038210340600926e-01 6.58033176041716849624663154827431e-02 4.02585998992950350877650933512086e-03
            1.26834046769924602138246427784907e-01 8.02927161901378694253139656211715e-01 7.02387913286967036086139160033781e-02 3.70589453915654140278568284827543e-03
            8.04415140888906643468203583324794e-02 8.45587809011131841252506546879886e-01 7.39706768999774944006730947876349e-02 3.18857626274490985399112652487474e-03
            4.38827858743368603988699305773480e-02 8.79205697774942618316629250330152e-01 7.69115163507204657733495878346730e-02 2.49504713671733430527255315212187e-03
            1.80140363610431508689657675859053e-02 9.02993525909800864859278135554632e-01 7.89924377291559842717560968594626e-02 1.66012326601700107263359651454948e-03
            3.43570040745255766978516476228833e-03 9.16399158446388084620082281617215e-01 8.01651411461593577101325536204968e-02 7.30898753882136412510228673511392e-04
            9.96564299592547442330214835237712e-01 2.99993662128627104970779448933627e-03 4.35763786166286620077370272952066e-04 3.08422457640215244505208121161832e-06
            9.81985963638956738108731769898441e-01 1.57292432307115787049056621071941e-02 2.28479313033168318636256799436524e-03 3.72756222392179873879013152215833e-05
            9.56117214125662862045373913133517e-01 3.83169545583568749158587252168218e-02 5.56583131598026303876736164966132e-03 1.40165334864363760620370458198636e-04
            9.19558485911109335653179641667521e-01 7.02387913286967591197651472612051e-02 1.02027227601939052270552110712742e-02 3.41410205792746287035283048183487e-04
            8.73165953230075397861753572215093e-01 1.10747171349889178548764334664156e-01 1.60868754200354235894820931207505e-02 6.58886975968105180825884481521371e-04
            8.18026840363257568000676656083670e-01 1.58892767396504847043914310233959e-01 2.30803922402375849554090336823720e-02 1.09616835446737039862219464936288e-03
            7.55433500975413618760967437992804e-01 2.13547140248945277463121783512179e-01 3.10193587756411037759107784950174e-02 1.64141279003868971290980294241990e-03
            6.86853044357709663358946272637695e-01 2.73429260024496578740382801697706e-01 3.97176956177937579006709256645991e-02 2.26779043813519478808848894857420e-03
            6.13892925570822534098169853677973e-01 3.37135551692828439040994226161274e-01 4.89715227363490268608359201607527e-02 2.93542157938367440778160144532194e-03
            5.38263260566748669155856532597682e-01 4.03172800228581817982842494529905e-01 5.85639392046695128613009728724137e-02 3.59466000715142957808168056033082e-03
            4.61736739433251219821841004886664e-01 4.69993153001493635390062308943016e-01 6.82701075652551447880966861703200e-02 4.19042551921064172076869525085385e-03
            3.86107074429177521412981377579854e-01 5.36030401537246903309608114795992e-01 7.78625240335756307885617388819810e-02 4.66718861294032096032058376522400e-03
            3.13146955642290225618751264846651e-01 5.99736693205578874632522001775214e-01 8.71163511521308997487267333781347e-02 4.97414628605799720439950561967635e-03
            2.44566499024586769817091180811985e-01 6.59618812981129787331724401155952e-01 9.58146879942833873400331867742352e-02 5.07010655780819850801943715623565e-03
            1.81973159636742376488172112658503e-01 7.14273185833570467551112415094394e-01 1.03753654529687100449564240989275e-01 4.92762304782270923325659950364752e-03
            1.26834046769924602138246427784907e-01 7.62418781880186191557413621922024e-01 1.10747171349889206304339950293070e-01 4.53598768701041210604785192117561e-03
            8.04415140888906643468203583324794e-02 8.02927161901378694253139656211715e-01 1.16631324009730641400039985455805e-01 3.90279391765865901753329580969876e-03
            4.38827858743368603988699305773480e-02 8.34848998671718689479348540771753e-01 1.21268215453944394610630297393072e-01 3.05391936307941157940915033464080e-03
            1.80140363610431508689657675859053e-02 8.57436709999363944056938180438010e-01 1.24549253639592905074096051976085e-01 2.03197867991314123567447857965362e-03
            3.43570040745255766978516476228833e-03 8.70166016608789116837385790859116e-01 1.26398282983758325492829044378595e-01 8.94614704501330818171844772024315e-04
            9.96564299592547442330214835237712e-01 2.81049514874317238680156982866265e-03 6.25205258709385282983594933625682e-04 3.57635683079427176055218456918183e-06
            9.81985963638956738108731769898441e-01 1.47359652466130530701349243827281e-02 3.27807111443020882113330571883125e-03 4.32234822448777751481736875138751e-05
            9.56117214125662862045373913133517e-01 3.58972966751213984348822805259260e-02 7.98548919921573951974380634055706e-03 1.62530723805947527891083148610107e-04
            9.19558485911109335653179641667521e-01 6.58033176041716988402541232971998e-02 1.46381964847189655065662350352795e-02 3.95887099445302943941105144887160e-04
            8.73165953230075397861753572215093e-01 1.03753654529687058816200817545905e-01 2.30803922402375433220456102390017e-02 7.64021840450330677087242836620362e-04
            8.18026840363257568000676656083670e-01 1.48858928808563051315871916813194e-01 3.31142308281793806834514271031367e-02 1.27107773285855879183414707256361e-03
            7.55433500975413618760967437992804e-01 2.00061960455786114199128178370302e-01 4.45045385688002670399043836368946e-02 1.90332373612553873085118105734637e-03
            6.86853044357709663358946272637695e-01 2.56162614693435930490750251919962e-01 5.69843409488544061503034754423425e-02 2.62964891930658750673899248795351e-03
            6.13892925570822534098169853677973e-01 3.15845950137201181284751783095999e-01 7.02611242919762846170783632260282e-02 3.40381018198619754791534042226431e-03
            5.38263260566748669155856532597682e-01 3.77713046038215316535513466078555e-01 8.40236933950360143086300013237633e-02 4.16823955341011267300421749837369e-03
            4.61736739433251219821841004886664e-01 4.40313794325042306976314421262941e-01 9.79494662417064732018445738503942e-02 4.85906799531628068311572832271850e-03
            3.86107074429177521412981377579854e-01 5.02180890226056275693622410472017e-01 1.11712035344766258404547443205956e-01 5.41190547672944482354706607907247e-03
            3.13146955642290225618751264846651e-01 5.61864225669821748532228866679361e-01 1.24988818687888025849019868473988e-01 5.76784264791291387958205305608317e-03
            2.44566499024586769817091180811985e-01 6.17964879907471065223489858908579e-01 1.37468621067942109448267729021609e-01 5.87911475694948958831131946567439e-03
            1.81973159636742376488172112658503e-01 6.69167911554694461173653508012649e-01 1.48858928808563106827023148071021e-01 5.71389595994263241318655843770102e-03
            1.26834046769924602138246427784907e-01 7.14273185833570467551112415094394e-01 1.58892767396504930310641157120699e-01 5.25976956183983428150252592558900e-03
            8.04415140888906643468203583324794e-02 7.52223522759085883038210340600926e-01 1.67334963152023452614969301066594e-01 4.52554064752414605260488755789083e-03
            4.38827858743368603988699305773480e-02 7.82129543688136363854823684960138e-01 1.73987670437526720235155153204687e-01 3.54121598615386853556308288659693e-03
            1.80140363610431508689657675859053e-02 8.03290875116644653708419809845509e-01 1.78695088522312195422614422568586e-01 2.35621001386772388475066009050352e-03
            3.43570040745255766978516476228833e-03 8.15216345214514359618362959736260e-01 1.81347954378033082711851875501452e-01 1.03736330805865271942334260302232e-03
            9.96564299592547442330214835237712e-01 2.59544318710454085738503771096930e-03 8.40257220348016812400127051319032e-04 3.98466455122515066691575927326774e-06
            9.81985963638956738108731769898441e-01 1.36084065549213120011051714186578e-02 4.40562980612194989016305868290146e-03 4.81582475212417252929258948412894e-05
            9.56117214125662862045373913133517e-01 3.31505265656049305511565705728572e-02 1.07322593087322074034695162936259e-02 1.81086632088301114013387249634945e-04
            9.19558485911109335653179641667521e-01 6.07682146119337338041610507843870e-02 1.96732994769569305426593075480923e-02 4.41084982869761172384615033337241e-04
            8.73165953230075397861753572215093e-01 9.58146879942834983623356492898893e-02 3.10193587756411037759107784950174e-02 8.51249158862066724988426624776139e-04
            8.18026840363257568000676656083670e-01 1.37468621067942109448267729021609e-01 4.45045385688003225510556148947217e-02 1.41619492226347214874904167913883e-03
            7.55433500975413618760967437992804e-01 1.84753726579443361410781676568149e-01 5.98127724451430198282508854390471e-02 2.12062357859310558413912239927868e-03
            6.86853044357709663358946272637695e-01 2.36561701020647940918451013203594e-01 7.65852546216423957226027141587110e-02 2.92987230488445755602344711121532e-03
            6.13892925570822534098169853677973e-01 2.91678218987408144347028837728431e-01 9.44288554417693215548013085935963e-02 3.79241848980920023479135316790689e-03
            5.38263260566748669155856532597682e-01 3.48811401599033399367755237108213e-01 1.12925337834217931476388230294106e-01 4.64412170689330622980195428795014e-03
            4.61736739433251219821841004886664e-01 4.06622099376380330415514663400245e-01 1.31641161190368449762644331713091e-01 5.41382107797934528786720775883623e-03
            3.86107074429177521412981377579854e-01 4.63755281988005418902787369006546e-01 1.50137643582817115195382484671427e-01 6.02977525117810633087378491268282e-03
            3.13146955642290225618751264846651e-01 5.18871799954765733353667656047037e-01 1.67981244402944041027581079106312e-01 6.42634927764714359438302793137154e-03
            2.44566499024586769817091180811985e-01 5.70679774395969952038853989506606e-01 1.84753726579443222632903598423582e-01 6.55032516970588743476788806674449e-03
            1.81973159636742376488172112658503e-01 6.17964879907471509312699708971195e-01 2.00061960455786058687976947112475e-01 6.36624357081155226728563789606596e-03
            1.26834046769924602138246427784907e-01 6.59618812981130120398631788702914e-01 2.13547140248945277463121783512179e-01 5.86027018898491267734351950480232e-03
            8.04415140888906643468203583324794e-02 6.94665286363479905773488098930102e-01 2.24893199547629429879691542737419e-01 5.04221537349031512925678910619354e-03
            4.38827858743368603988699305773480e-02 7.22282974409808820048795041657286e-01 2.33834239715854264041183796507539e-01 3.94551172488158147927395091869585e-03
            1.80140363610431508689657675859053e-02 7.41825094420492381352971733576851e-01 2.40160869218464467778062498837244e-01 2.62521525723016443953916621012468e-03
            3.43570040745255766978516476228833e-03 7.52838057788309034101814631867455e-01 2.43726241804238408228400203370256e-01 1.15579764434326631716842825881031e-03
            9.96564299592547442330214835237712e-01 2.35982128435981283395839191996401e-03 1.07587912309274483582677284232432e-03 4.29957615429740088022541807943888e-06
            9.81985963638956738108731769898441e-01 1.23729957157530431194336628664132e-02 5.64104064529021877183456723514610e-03 5.19642368920161275410926626783237e-05
            9.56117214125662862045373913133517e-01 3.01410250726859597902862475393704e-02 1.37417608016511781643398393271127e-02 1.95398070572714584831539030851388e-04
            9.19558485911109335653179641667521e-01 5.52514988446981478165120904577634e-02 2.51900152441925165303082678747160e-02 4.75944323539706590914244443979442e-04
            8.73165953230075397861753572215093e-01 8.71163511521308442375755021203076e-02 3.97176956177937579006709256645991e-02 9.18524141181152153738664889459642e-04
            8.18026840363257568000676656083670e-01 1.24988818687887998093444252845075e-01 5.69843409488544339058790910712560e-02 1.52811807350982948455753174243910e-03
            7.55433500975413618760967437992804e-01 1.67981244402943985516429847848485e-01 7.65852546216423957226027141587110e-02 2.28821835653802421392533084087972e-03
            6.86853044357709663358946272637695e-01 2.15085939814255777680074288582546e-01 9.80610158280345589609794387797592e-02 3.16142273340032104594898854088569e-03
            6.13892925570822534098169853677973e-01 2.65198819519729345195457881345646e-01 1.20908254909448120706372264976380e-01 4.09213671471710636884377976230098e-03
            5.38263260566748669155856532597682e-01 3.17145285171531232659702936871327e-01 1.44591454261720098184440530530992e-01 5.01115079874766070067160228518333e-03
            4.61736739433251219821841004886664e-01 3.69707759186178541721545798282023e-01 1.68555501380570238456613196831313e-01 5.84168019949275404978372705500078e-03
            3.86107074429177521412981377579854e-01 4.21654224837980262652337160034222e-01 1.92238700732842271445832693643752e-01 6.50631378186321794598878653914653e-03
            3.13146955642290225618751264846651e-01 4.71767104543453941190023215312976e-01 2.15085939814255833191225519840373e-01 6.93422940831054691879220541750328e-03
            2.44566499024586769817091180811985e-01 5.18871799954765400286760268500075e-01 2.36561701020647774384997319430113e-01 7.06800322599355178299074253800427e-03
            1.81973159636742376488172112658503e-01 5.61864225669821637509926404163707e-01 2.56162614693435930490750251919962e-01 6.86937349370964501354963616108762e-03
            1.26834046769924602138246427784907e-01 5.99736693205578763610219539259560e-01 2.73429260024496634251534032955533e-01 6.32341258301211909392813836916503e-03
            8.04415140888906643468203583324794e-02 6.31601545513011508603540278272703e-01 2.87956940398097827049639363394817e-01 5.44070616384132940690498969615874e-03
            4.38827858743368603988699305773480e-02 6.56712019285023873571560670825420e-01 2.99405194840639210518418167339405e-01 4.25732904507251062875550218222998e-03
            1.80140363610431508689657675859053e-02 6.74480048641956653199258653330617e-01 3.07505914997000195931775579083478e-01 2.83268836680720427489976742663202e-03
            3.43570040745255766978516476228833e-03 6.84493223073349810192667064256966e-01 3.12071076519197632137547770980746e-01 1.24714136583554588942612806334864e-03
            9.96564299592547442330214835237712e-01 2.10915217451591776448571735613768e-03 1.32654823293663990529944740615065e-03 4.51370990085204018028852690491703e-06
            9.81985963638956738108731769898441e-01 1.10586894830200215683602138483366e-02 6.95534687802324032290801625322274e-03 5.45522354140142936868712597586750e-05
            9.56117214125662862045373913133517e-01 2.69393318025947896277028092981709e-02 1.69434540717423483269232775683122e-02 2.05129569543715026659971845113262e-04
            9.19558485911109335653179641667521e-01 4.93824764213756303710134432094492e-02 3.10590376675150339758069151230302e-02 4.99647995132802206878774864406978e-04
            8.73165953230075397861753572215093e-01 7.78625240335755475218348919952405e-02 4.89715227363490546164115357896662e-02 9.64269816706730476321696166053243e-04
            8.18026840363257568000676656083670e-01 1.11712035344766091871093749432475e-01 7.02611242919763401282295944838552e-02 1.60422363287559665429027333516387e-03
            7.55433500975413618760967437992804e-01 1.50137643582817059684231253413600e-01 9.44288554417693215548013085935963e-02 2.40217953597447872113312961062093e-03
            6.86853044357709663358946272637695e-01 1.92238700732842215934681462385925e-01 1.20908254909448120706372264976380e-01 3.31887250752966028038981605163826e-03
            6.13892925570822534098169853677973e-01 2.37028401504919078446320668263070e-01 1.49078672924258387455509478058957e-01 4.29593926052394044917681625861405e-03
            5.38263260566748669155856532597682e-01 2.83456917814211228900944661290850e-01 1.78279821619040101943198806111468e-01 5.26072342092661574680656499936049e-03
            4.61736739433251219821841004886664e-01 3.30436007756611360708376423644950e-01 2.07827252810137419469782571468386e-01 6.13261605512149289565204668406295e-03
            3.86107074429177521412981377579854e-01 3.76864524065903372385122338528163e-01 2.37028401504919161713047515149810e-01 6.83035068605386454926309269808371e-03
            3.13146955642290225618751264846651e-01 4.21654224837980373674639622549876e-01 2.65198819519729400706609112603473e-01 7.27957798905072041295571949603982e-03
            2.44566499024586769817091180811985e-01 4.63755281988005196858182443975238e-01 2.91678218987407977813575143954949e-01 7.42001420501285473274588611047875e-03
            1.81973159636742376488172112658503e-01 5.02180890226056386715924872987671e-01 3.15845950137201181284751783095999e-01 7.21149202584006436544949991684916e-03
            1.26834046769924602138246427784907e-01 5.36030401537247014331910577311646e-01 3.37135551692828383529842994903447e-01 6.63834037561738827398771789489729e-03
            8.04415140888906643468203583324794e-02 5.64510449149446924543838122190209e-01 3.55048036761662411109341519477312e-01 5.71167212721932805730151727630073e-03
            4.38827858743368603988699305773480e-02 5.86953593768227865901110362756299e-01 3.69163620357435218188868475408526e-01 4.46935873963347045367822119033008e-03
            1.80140363610431508689657675859053e-02 6.02834236087802621817388626368484e-01 3.79151727551154227313645606045611e-01 2.97376603377675014297509825667021e-03
            3.43570040745255766978516476228833e-03 6.11783773396306629344110206147889e-01 3.84780526196240812986104629089823e-01 1.30925331444767018181096762674542e-03
            9.96564299592547442330214835237712e-01 1.84931130364592065731810244244571e-03 1.58638910380663701246706231984263e-03 4.62204647224965192947674502699229e-06
            9.81985963638956738108731769898441e-01 9.69629394766311457354301950317677e-03 8.31774241338014731772521059838255e-03 5.58615801164095599775326339564430e-05
            9.56117214125662862045373913133517e-01 2.36204914074731701922882365352052e-02 2.02622944668639677623378503312779e-02 2.10053021591982274100460226229359e-04
            9.19558485911109335653179641667521e-01 4.32987116584123421514362917150720e-02 3.71428024304783221953840666174074e-02 5.11640380972255641965629191503240e-04
            8.73165953230075397861753572215093e-01 6.82701075652551170325210705414065e-02 5.85639392046694851057253572435002e-02 9.87413901758472717026804410522800e-04
            8.18026840363257568000676656083670e-01 9.79494662417064176906933425925672e-02 8.40236933950360143086300013237633e-02 1.64272767765436195422368292895499e-03
            7.55433500975413618760967437992804e-01 1.31641161190368422007068716084177e-01 1.12925337834217959231963845923019e-01 2.45983585428590943375293242922908e-03
            6.86853044357709663358946272637695e-01 1.68555501380570210701037581202399e-01 1.44591454261720125940016146159905e-01 3.39853098719927445084398875962961e-03
            6.13892925570822534098169853677973e-01 2.07827252810137363958631340210559e-01 1.78279821619040101943198806111468e-01 4.39904897307540499601374861526892e-03
            5.38263260566748669155856532597682e-01 2.48535922890801092011159312278323e-01 2.13200816542450238832984155123995e-01 5.38698956363701120936271493633285e-03
            4.61736739433251219821841004886664e-01 2.89727337675947604900272835948272e-01 2.48535922890801175277886159165064e-01 6.27980907631773676430553265959134e-03
            3.86107074429177521412981377579854e-01 3.30436007756611249686073961129296e-01 2.83456917814211284412095892548678e-01 6.99429050297269584018922117252259e-03
            3.13146955642290225618751264846651e-01 3.69707759186178541721545798282023e-01 3.17145285171531232659702936871327e-01 7.45430001104119104027301290216201e-03
            2.44566499024586769817091180811985e-01 4.06622099376379997348607275853283e-01 3.48811401599033177323150312076905e-01 7.59810693058676159955533435663710e-03
            1.81973159636742376488172112658503e-01 4.40313794325042251465163190005114e-01 3.77713046038215316535513466078555e-01 7.38457987107204311127972573558509e-03
            1.26834046769924602138246427784907e-01 4.69993153001493579878911077685189e-01 4.03172800228581817982842494529905e-01 6.79767162460371936194558628585582e-03
            8.04415140888906643468203583324794e-02 4.94964548908336354759995856511523e-01 4.24593937002772980893183785155998e-01 5.84876179155363520173471769680873e-03
            4.38827858743368603988699305773480e-02 5.14642769159275648149787230067886e-01 4.41474444966387435940191608096939e-01 4.57663080913586414744109731600474e-03
            1.80140363610431508689657675859053e-02 5.28566966619085643053210787911667e-01 4.53418997019871206077823444502428e-01 3.04514138206336305858545898672674e-03
            3.43570040745255766978516476228833e-03 5.36413949263102707298855875706067e-01 4.60150350329444735031358959531644e-01 1.34067757925286851553026146177672e-03
            9.96564299592547442330214835237712e-01 1.58638910380663657878619332564085e-03 1.84931130364592109099897143664748e-03 4.62204647224962821255422190658280e-06
            9.81985963638956738108731769898441e-01 8.31774241338014558300173462157545e-03 9.69629394766311630826649547998386e-03 5.58615801164092685981987784771263e-05
            9.56117214125662862045373913133517e-01 2.02622944668639642928908983776637e-02 2.36204914074731736617351884888194e-02 2.10053021591981162793233428587314e-04
            9.19558485911109335653179641667521e-01 3.71428024304783152564901627101790e-02 4.32987116584123490903301956223004e-02 5.11640380972252931460197977742155e-04
            8.73165953230075397861753572215093e-01 5.85639392046694712279375494290434e-02 6.82701075652551309103088783558633e-02 9.87413901758467512856376480101517e-04
            8.18026840363257568000676656083670e-01 8.40236933950359449196909622514795e-02 9.79494662417064870796323816648510e-02 1.64272767765435328060630304491951e-03
            7.55433500975413618760967437992804e-01 1.12925337834217931476388230294106e-01 1.31641161190368449762644331713091e-01 2.45983585428589642332686260317587e-03
            6.86853044357709663358946272637695e-01 1.44591454261720098184440530530992e-01 1.68555501380570238456613196831313e-01 3.39853098719925623624749100315512e-03
            6.13892925570822534098169853677973e-01 1.78279821619040046432047574853641e-01 2.07827252810137419469782571468386e-01 4.39904897307538157724682292837315e-03
            5.38263260566748669155856532597682e-01 2.13200816542450183321832923866168e-01 2.48535922890801147522310543536150e-01 5.38698956363698258642536131901579e-03
            4.61736739433251219821841004886664e-01 2.48535922890801092011159312278323e-01 2.89727337675947715922575298463926e-01 6.27980907631770380455948910025654e-03
            3.86107074429177521412981377579854e-01 2.83456917814211117878642198775196e-01 3.30436007756611416219527654902777e-01 6.99429050297265854363448767117006e-03
            3.13146955642290225618751264846651e-01 3.17145285171531177148551705613500e-01 3.69707759186178597232697029539850e-01 7.45430001104115200899480342400238e-03
            2.44566499024586769817091180811985e-01 3.48811401599033066300847849561251e-01 4.06622099376380108370909738368937e-01 7.59810693058672170091538689007393e-03
            1.81973159636742376488172112658503e-01 3.77713046038215261024362234820728e-01 4.40313794325042306976314421262941e-01 7.38457987107200494736325424582901e-03
            1.26834046769924602138246427784907e-01 4.03172800228581762471691263272078e-01 4.69993153001493635390062308943016e-01 6.79767162460368293275259077290684e-03
            8.04415140888906643468203583324794e-02 4.24593937002772925382032553898171e-01 4.94964548908336410271147087769350e-01 5.84876179155360484407388810268458e-03
            4.38827858743368603988699305773480e-02 4.41474444966387380429040376839112e-01 5.14642769159275648149787230067886e-01 4.57663080913583986131243364070542e-03
            1.80140363610431508689657675859053e-02 4.53418997019871150566672213244601e-01 5.28566966619085754075513250427321e-01 3.04514138206334701239330620126111e-03
            3.43570040745255766978516476228833e-03 4.60150350329444568497905265758163e-01 5.36413949263102818321158338221721e-01 1.34067757925286135979592305744745e-03
            9.96564299592547442330214835237712e-01 1.32654823293664012213988190325153e-03 2.10915217451591776448571735613768e-03 4.51370990085201646336600378450754e-06
            9.81985963638956738108731769898441e-01 6.95534687802324119026975424162629e-03 1.10586894830200198336367378715295e-02 5.45522354140139955312738262449557e-05
            9.56117214125662862045373913133517e-01 1.69434540717423483269232775683122e-02 2.69393318025947896277028092981709e-02 2.05129569543713915352745047471217e-04
            9.19558485911109335653179641667521e-01 3.10590376675150409147008190302586e-02 4.93824764213756234321195393022208e-02 4.99647995132799496373343650645893e-04
            8.73165953230075397861753572215093e-01 4.89715227363490615553054396968946e-02 7.78625240335755475218348919952405e-02 9.64269816706725272151268235631960e-04
            8.18026840363257568000676656083670e-01 7.02611242919762846170783632260282e-02 1.11712035344766147382244980690302e-01 1.60422363287558798067289345112840e-03
            7.55433500975413618760967437992804e-01 9.44288554417693493103769242225098e-02 1.50137643582817031928655637784686e-01 2.40217953597446571070705978456772e-03
            6.86853044357709663358946272637695e-01 1.20908254909448148461947880605294e-01 1.92238700732842188179105846757011e-01 3.31887250752964249947418728936555e-03
            6.13892925570822534098169853677973e-01 1.49078672924258415211085093687871e-01 2.37028401504919050690745052634156e-01 4.29593926052391703040989057171828e-03
            5.38263260566748669155856532597682e-01 1.78279821619040129698774421740382e-01 2.83456917814211228900944661290850e-01 5.26072342092658799123094937044698e-03
            4.61736739433251219821841004886664e-01 2.07827252810137447225358187097299e-01 3.30436007756611305197225192387123e-01 6.13261605512145993590600312472816e-03
            3.86107074429177521412981377579854e-01 2.37028401504919078446320668263070e-01 3.76864524065903427896273569785990e-01 6.83035068605382725270835919673118e-03
            3.13146955642290225618751264846651e-01 2.65198819519729400706609112603473e-01 4.21654224837980373674639622549876e-01 7.27957798905068051431577202947665e-03
            2.44566499024586769817091180811985e-01 2.91678218987407977813575143954949e-01 4.63755281988005196858182443975238e-01 7.42001420501281483410593864391558e-03
            1.81973159636742376488172112658503e-01 3.15845950137201181284751783095999e-01 5.02180890226056386715924872987671e-01 7.21149202584002533417129043868954e-03
            1.26834046769924602138246427784907e-01 3.37135551692828439040994226161274e-01 5.36030401537247014331910577311646e-01 6.63834037561735271215646037035185e-03
            8.04415140888906643468203583324794e-02 3.55048036761662466620492750735139e-01 5.64510449149446813521535659674555e-01 5.71167212721929683227894969377303e-03
            4.38827858743368603988699305773480e-02 3.69163620357435273700019706666353e-01 5.86953593768227754878807900240645e-01 4.46935873963344616754955751503076e-03
            1.80140363610431508689657675859053e-02 3.79151727551154338335948068561265e-01 6.02834236087802510795086163852829e-01 2.97376603377673409678294547120458e-03
            3.43570040745255766978516476228833e-03 3.84780526196240868497255860347650e-01 6.11783773396306518321807743632235e-01 1.30925331444766302607662922241616e-03
            9.96564299592547442330214835237712e-01 1.07587912309274461898633834522343e-03 2.35982128435981326763926091416579e-03 4.29957615429741189165373238534329e-06
            9.81985963638956738108731769898441e-01 5.64104064529021790447282924674255e-03 1.23729957157530448541571388432203e-02 5.19642368920162630663642233663779e-05
            9.56117214125662862045373913133517e-01 1.37417608016511712254459354198843e-02 3.01410250726859667291801514465988e-02 1.95398070572715099827570961465995e-04
            9.19558485911109335653179641667521e-01 2.51900152441925095914143639674876e-02 5.52514988446981547554059943649918e-02 4.75944323539707783536634178034319e-04
            8.73165953230075397861753572215093e-01 3.97176956177937440228831178501423e-02 8.71163511521308581153633099347644e-02 9.18524141181154538983444357569397e-04
            8.18026840363257568000676656083670e-01 5.69843409488543645169400519989722e-02 1.24988818687888067482383291917358e-01 1.52811807350983338768535269025506e-03
            7.55433500975413618760967437992804e-01 7.65852546216423679670270985297975e-02 1.67981244402944013272005463477399e-01 2.28821835653803028545749675970455e-03
            6.86853044357709663358946272637695e-01 9.80610158280345173276160153363890e-02 2.15085939814255833191225519840373e-01 3.16142273340032885220463043651762e-03
            6.13892925570822534098169853677973e-01 1.20908254909448092950796649347467e-01 2.65198819519729345195457881345646e-01 4.09213671471711677718463562314355e-03
            5.38263260566748669155856532597682e-01 1.44591454261720070428864914902078e-01 3.17145285171531288170854168129154e-01 5.01115079874767371109767211123653e-03
            4.61736739433251219821841004886664e-01 1.68555501380570182945461965573486e-01 3.69707759186178597232697029539850e-01 5.84168019949276879493327285786108e-03
            3.86107074429177521412981377579854e-01 1.92238700732842104912378999870270e-01 4.21654224837980429185790853807703e-01 6.50631378186323442586180831881393e-03
            3.13146955642290225618751264846651e-01 2.15085939814255749924498672953632e-01 4.71767104543454052212325677828630e-01 6.93422940831056513338870317397777e-03
            2.44566499024586769817091180811985e-01 2.36561701020647718873846088172286e-01 5.18871799954765400286760268500075e-01 7.06800322599356999758724029447876e-03
            1.81973159636742376488172112658503e-01 2.56162614693435819468447789404308e-01 5.61864225669821748532228866679361e-01 6.86937349370966236078439592915856e-03
            1.26834046769924602138246427784907e-01 2.73429260024496467718080339182052e-01 5.99736693205578985654824464290868e-01 6.32341258301213470643942216042888e-03
            8.04415140888906643468203583324794e-02 2.87956940398097716027336900879163e-01 6.31601545513011619625842740788357e-01 5.44070616384134328469279751061549e-03
            4.38827858743368603988699305773480e-02 2.99405194840639099496115704823751e-01 6.56712019285023984593863133341074e-01 4.25732904507252190445809603147609e-03
            1.80140363610431508689657675859053e-02 3.07505914997000029398321885309997e-01 6.74480048641956875243863578361925e-01 2.83268836680721121379367133386040e-03
            3.43570040745255766978516476228833e-03 3.12071076519197465604094077207264e-01 6.84493223073350032237271989288274e-01 1.24714136583554892519221102276106e-03
            9.96564299592547442330214835237712e-01 8.40257220348018330283168531025240e-04 2.59544318710453955634243072836398e-03 3.98466455122514897284986476466706e-06
            9.81985963638956738108731769898441e-01 4.40562980612195769641870057853339e-03 1.36084065549213033274877915346224e-02 4.81582475212417049641351607380813e-05
            9.56117214125662862045373913133517e-01 1.07322593087322230159808000848898e-02 3.31505265656049166733687627584004e-02 1.81086632088301032698224313222113e-04
            9.19558485911109335653179641667521e-01 1.96732994769569617676818751306200e-02 6.07682146119336991096915312482452e-02 4.41084982869760955544180536236354e-04
            8.73165953230075397861753572215093e-01 3.10193587756411592870620097528445e-02 9.58146879942834428511844180320622e-02 8.51249158862066399727774879124809e-04
            8.18026840363257568000676656083670e-01 4.45045385688003294899495188019500e-02 1.37468621067942109448267729021609e-01 1.41619492226347149822773818783617e-03
            7.55433500975413618760967437992804e-01 5.98127724451431030949777323257877e-02 1.84753726579443278144054829681409e-01 2.12062357859310471677738441087513e-03
            6.86853044357709663358946272637695e-01 7.65852546216425206226929844888218e-02 2.36561701020647829896148550687940e-01 2.92987230488445625498084012861000e-03
            6.13892925570822534098169853677973e-01 9.44288554417694880882550023670774e-02 2.91678218987407977813575143954949e-01 3.79241848980919850006787719109980e-03
            5.38263260566748669155856532597682e-01 1.12925337834218139643205347510957e-01 3.48811401599033177323150312076905e-01 4.64412170689330449507847831114304e-03
            4.61736739433251219821841004886664e-01 1.31641161190368671807249256744399e-01 4.06622099376380108370909738368937e-01 5.41382107797934268578199379362559e-03
            3.86107074429177521412981377579854e-01 1.50137643582817253973260562815994e-01 4.63755281988005307880484906490892e-01 6.02977525117810372878857094747218e-03
            3.13146955642290225618751264846651e-01 1.67981244402944290827761619766534e-01 5.18871799954765511309062731015729e-01 6.42634927764714099229781396616090e-03
            2.44566499024586769817091180811985e-01 1.84753726579443555699810985970544e-01 5.70679774395969618971946601959644e-01 6.55032516970588483268267410153385e-03
            1.81973159636742376488172112658503e-01 2.00061960455786419510459950288350e-01 6.17964879907471176245792321424233e-01 6.36624357081154966520042393085532e-03
            1.26834046769924602138246427784907e-01 2.13547140248945610530029171059141e-01 6.59618812981129787331724401155952e-01 5.86027018898491007525830553959167e-03
            8.04415140888906643468203583324794e-02 2.24893199547629818457750161542208e-01 6.94665286363479461684278248867486e-01 5.04221537349031252717157514098290e-03
            4.38827858743368603988699305773480e-02 2.33834239715854597108091184054501e-01 7.22282974409808486981887654110324e-01 3.94551172488157974455047494188875e-03
            1.80140363610431508689657675859053e-02 2.40160869218464828600545502013119e-01 7.41825094420492048286064346029889e-01 2.62521525723016313849655922751936e-03
            3.43570040745255766978516476228833e-03 2.43726241804238741295307590917218e-01 7.52838057788308701034907244320493e-01 1.15579764434326588348755926460854e-03
            9.96564299592547442330214835237712e-01 6.25205258709385066143160436524795e-04 2.81049514874317282048243882286442e-03 3.57635683079426625483802741622963e-06
            9.81985963638956738108731769898441e-01 3.27807111443020708640982974202416e-03 1.47359652466130548048584003595352e-02 4.32234822448777073855379071698479e-05
            9.56117214125662862045373913133517e-01 7.98548919921573605029685438694287e-03 3.58972966751213984348822805259260e-02 1.62530723805947283945594339371610e-04
            9.19558485911109335653179641667521e-01 1.46381964847189620371192830816653e-02 6.58033176041716988402541232971998e-02 3.95887099445302347629910277859722e-04
            8.73165953230075397861753572215093e-01 2.30803922402375398525986582853875e-02 1.03753654529687058816200817545905e-01 7.64021840450329592885070351115928e-04
            8.18026840363257568000676656083670e-01 3.31142308281793251723001958453096e-02 1.48858928808563106827023148071021e-01 1.27107773285855684027023659865563e-03
            7.55433500975413618760967437992804e-01 4.45045385688002531621165758224379e-02 2.00061960455786114199128178370302e-01 1.90332373612553591192553259503484e-03
            6.86853044357709663358946272637695e-01 5.69843409488543853336217637206573e-02 2.56162614693435930490750251919962e-01 2.62964891930658360361117154013755e-03
            6.13892925570822534098169853677973e-01 7.02611242919762846170783632260282e-02 3.15845950137201181284751783095999e-01 3.40381018198619234374491249184302e-03
            5.38263260566748669155856532597682e-01 8.40236933950359587974787700659363e-02 3.77713046038215372046664697336382e-01 4.16823955341010573411031359114531e-03
            4.61736739433251219821841004886664e-01 9.79494662417064315684811504070240e-02 4.40313794325042362487465652520768e-01 4.85906799531627287686008642708657e-03
            3.86107074429177521412981377579854e-01 1.11712035344766077993305941618019e-01 5.02180890226056497738227335503325e-01 5.41190547672943614992968619503699e-03
            3.13146955642290225618751264846651e-01 1.24988818687888011971232060659531e-01 5.61864225669821748532228866679361e-01 5.76784264791290520596467317204770e-03
            2.44566499024586769817091180811985e-01 1.37468621067942026181540882134868e-01 6.17964879907471176245792321424233e-01 5.87911475694948091469393958163892e-03
            1.81973159636742376488172112658503e-01 1.48858928808563051315871916813194e-01 6.69167911554694461173653508012649e-01 5.71389595994262373956917855366555e-03
            1.26834046769924602138246427784907e-01 1.58892767396504847043914310233959e-01 7.14273185833570578573414877610048e-01 5.25976956183982560788514604155353e-03
            8.04415140888906643468203583324794e-02 1.67334963152023424859393685437681e-01 7.52223522759085883038210340600926e-01 4.52554064752413911371098365066246e-03
            4.38827858743368603988699305773480e-02 1.73987670437526692479579537575773e-01 7.82129543688136363854823684960138e-01 3.54121598615386289771178596197387e-03
            1.80140363610431508689657675859053e-02 1.78695088522312195422614422568586e-01 8.03290875116644653708419809845509e-01 2.35621001386772041530370813688933e-03
            3.43570040745255766978516476228833e-03 1.81347954378032999445125028614711e-01 8.15216345214514470640665422251914e-01 1.03736330805865098469986662621523e-03
            9.96564299592547442330214835237712e-01 4.35763786166286728497587521502510e-04 2.99993662128627104970779448933627e-03 3.08422457640219776131475931668646e-06
            9.81985963638956738108731769898441e-01 2.28479313033168362004343698856701e-03 1.57292432307115787049056621071941e-02 3.72756222392185294889875579738003e-05
            9.56117214125662862045373913133517e-01 5.56583131598026303876736164966132e-03 3.83169545583568749158587252168218e-02 1.40165334864365820604498180657060e-04
            9.19558485911109335653179641667521e-01 1.02027227601939069617786870480813e-02 7.02387913286967591197651472612051e-02 3.41410205792751328575385105779105e-04
            8.73165953230075397861753572215093e-01 1.60868754200354201200351411671363e-02 1.10747171349889178548764334664156e-01 6.58886975968114830225219602510833e-04
            8.18026840363257568000676656083670e-01 2.30803922402375398525986582853875e-02 1.58892767396504902555065541491786e-01 1.09616835446738644481434743482851e-03
            7.55433500975413618760967437992804e-01 3.10193587756411072453577304486316e-02 2.13547140248945277463121783512179e-01 1.64141279003871378219803212061834e-03
            6.86853044357709663358946272637695e-01 3.97176956177937579006709256645991e-02 2.73429260024496578740382801697706e-01 2.26779043813522818151540150211076e-03
            6.13892925570822534098169853677973e-01 4.89715227363490546164115357896662e-02 3.37135551692828383529842994903447e-01 2.93542157938371734218763187129753e-03
            5.38263260566748669155856532597682e-01 5.85639392046694920446192611507286e-02 4.03172800228581817982842494529905e-01 3.59466000715148248714769785294720e-03
            4.61736739433251219821841004886664e-01 6.82701075652551309103088783558633e-02 4.69993153001493635390062308943016e-01 4.19042551921070330345209242750570e-03
            3.86107074429177521412981377579854e-01 7.78625240335755336440470841807837e-02 5.36030401537247014331910577311646e-01 4.66718861294038948189788484910423e-03
            3.13146955642290225618751264846651e-01 8.71163511521308581153633099347644e-02 5.99736693205578874632522001775214e-01 4.97414628605807093014723463397786e-03
            2.44566499024586769817091180811985e-01 9.58146879942834428511844180320622e-02 6.59618812981129787331724401155952e-01 5.07010655780827310112890415894071e-03
            1.81973159636742376488172112658503e-01 1.03753654529687058816200817545905e-01 7.14273185833570467551112415094394e-01 4.92762304782278209164259052954549e-03
            1.26834046769924602138246427784907e-01 1.10747171349889178548764334664156e-01 7.62418781880186191557413621922024e-01 4.53598768701047889290167702824874e-03
            8.04415140888906643468203583324794e-02 1.16631324009730696911191216713632e-01 8.02927161901378694253139656211715e-01 3.90279391765871626340800304433287e-03
            4.38827858743368603988699305773480e-02 1.21268215453944366855054681764159e-01 8.34848998671718689479348540771753e-01 3.05391936307945624853865673742348e-03
            1.80140363610431508689657675859053e-02 1.24549253639592932829671667604998e-01 8.57436709999363944056938180438010e-01 2.03197867991317115965443917957600e-03
            3.43570040745255766978516476228833e-03 1.26398282983758325492829044378595e-01 8.70166016608789116837385790859116e-01 8.94614704501343937018131846627966e-04
            9.96564299592547442330214835237712e-01 2.76372942731302299879114769964872e-04 3.15932746472125536990604999232346e-03 2.51980644655458326292463318296644e-06
            9.81985963638956738108731769898441e-01 1.44907635973465036792473537019532e-03 1.65649600013086130412265362110702e-02 3.04541225487815068919805056024686e-05
            9.56117214125662862045373913133517e-01 3.52999773817026319192025063387064e-03 4.03527881361668769311101812036213e-02 1.14514849883828062648317980265489e-04
            9.19558485911109335653179641667521e-01 6.47083718891319509963766520854733e-03 7.39706768999774666450974791587214e-02 2.78931581071713806385670153531464e-04
            8.73165953230075397861753572215093e-01 1.02027227601939069617786870480813e-02 1.16631324009730696911191216713632e-01 5.38309584294942018549423590201286e-04
            8.18026840363257568000676656083670e-01 1.46381964847189620371192830816653e-02 1.67334963152023480370544916695508e-01 8.95567757039964352887284615434282e-04
            7.55433500975413618760967437992804e-01 1.96732994769569305426593075480923e-02 2.24893199547629457635267158366332e-01 1.34103157125524982511144411745363e-03
            6.86853044357709663358946272637695e-01 2.51900152441925165303082678747160e-02 2.87956940398097827049639363394817e-01 1.85278108772284392201379343845247e-03
            6.13892925570822534098169853677973e-01 3.10590376675150339758069151230302e-02 3.55048036761662411109341519477312e-01 2.39823464078446017128110057115009e-03
            5.38263260566748669155856532597682e-01 3.71428024304783221953840666174074e-02 4.24593937002773036404335016413825e-01 2.93683136062627102302879933404256e-03
            4.61736739433251219821841004886664e-01 4.32987116584123490903301956223004e-02 4.94964548908336410271147087769350e-01 3.42357081189959867872918408693295e-03
            3.86107074429177521412981377579854e-01 4.93824764213756164932256353949924e-02 5.64510449149446924543838122190209e-01 3.81308548156764461414813816020342e-03
            3.13146955642290225618751264846651e-01 5.52514988446981547554059943649918e-02 6.31601545513011619625842740788357e-01 4.06386939965820009890151709441852e-03
            2.44566499024586769817091180811985e-01 6.07682146119336991096915312482452e-02 6.94665286363479461684278248867486e-01 4.14226878510481899553496987209655e-03
            1.81973159636742376488172112658503e-01 6.58033176041716988402541232971998e-02 7.52223522759085883038210340600926e-01 4.02585998992947488583915571780381e-03
            1.26834046769924602138246427784907e-01 7.02387913286967591197651472612051e-02 8.02927161901378694253139656211715e-01 3.70589453915651451457180520776546e-03
            8.04415140888906643468203583324794e-02 7.39706768999774666450974791587214e-02 8.45587809011131841252506546879886e-01 3.18857626274488686890506983218074e-03
            4.38827858743368603988699305773480e-02 7.69115163507204241399861643913027e-02 8.79205697774942618316629250330152e-01 2.49504713671731652435692438984915e-03
            1.80140363610431508689657675859053e-02 7.89924377291560259051195203028328e-02 9.02993525909800864859278135554632e-01 1.66012326601698914640969917400071e-03
            3.43570040745255766978516476228833e-03 8.01651411461593577101325536204968e-02 9.16399158446388084620082281617215e-01 7.30898753882131208339800743090109e-04
            9.96564299592547442330214835237712e-01 1.50768105308612487911595900769157e-04 3.28493230214394523608367748579440e-03 1.89634498686058047848483187197699e-06
            9.81985963638956738108731769898441e-01 7.90506100364179813654030848368848e-04 1.72235302606790832702365889872453e-02 2.29190312230469871851695068398769e-05
            9.56117214125662862045373913133517e-01 1.92569889609291080428965514670381e-03 4.19570869782442262829746937313757e-02 8.61810881527107105993731894777454e-05
            9.19558485911109335653179641667521e-01 3.52999773817024064051506293537841e-03 7.69115163507204241399861643913027e-02 2.09917117311010924555689238779621e-04
            8.73165953230075397861753572215093e-01 5.56583131598022834429784211351944e-03 1.21268215453944366855054681764159e-01 4.05118687966101180715494667339271e-04
            8.18026840363257568000676656083670e-01 7.98548919921568574331605105953713e-03 1.73987670437526747990730768833600e-01 6.73982491305576787363196267222065e-04
            7.55433500975413618760967437992804e-01 1.07322593087321380145304772213422e-02 2.33834239715854236285608180878626e-01 1.00922771304473406217550035535169e-03
            6.86853044357709663358946272637695e-01 1.37417608016510896934425645099509e-02 2.99405194840639266029569398597232e-01 1.39435794057017853168900511917627e-03
            6.13892925570822534098169853677973e-01 1.69434540717422407740677670062723e-02 3.69163620357435218188868475408526e-01 1.80485300550979480341262650711087e-03
            5.38263260566748669155856532597682e-01 2.02622944668638393928006280475529e-02 4.41474444966387491451342839354766e-01 2.21018778469813997542003924934306e-03
            4.61736739433251219821841004886664e-01 2.36204914074730244755162544834093e-02 5.14642769159275759172089692583540e-01 2.57649604602969151123659230506746e-03
            3.86107074429177521412981377579854e-01 2.69393318025946161553552116174615e-02 5.86953593768227865901110362756299e-01 2.86963530366737265206444895682125e-03
            3.13146955642290225618751264846651e-01 3.01410250726857759095977939978184e-02 6.56712019285023984593863133341074e-01 3.05836917507505429841829780457374e-03
            2.44566499024586769817091180811985e-01 3.31505265656047015676577416343207e-02 7.22282974409808486981887654110324e-01 3.11737064392514405877787631027331e-03
            1.81973159636742376488172112658503e-01 3.58972966751211763902773554946180e-02 7.82129543688136363854823684960138e-01 3.02976421865423306903353228847209e-03
            1.26834046769924602138246427784907e-01 3.83169545583566320545720884638285e-02 8.34848998671718800501651003287407e-01 2.78896600997771389532364771923767e-03
            8.04415140888906643468203583324794e-02 4.03527881361666201920357366361714e-02 8.79205697774942729338931712845806e-01 2.39964486929013448915370965153215e-03
            4.38827858743368603988699305773480e-02 4.19570869782439626050063452566974e-02 9.14160127147419121484972492908128e-01 1.87771173304371362736131523263339e-03
            1.80140363610431508689657675859053e-02 4.30922797739726887167321933702624e-02 9.38893683864984174292089846858289e-01 1.24936835421884310896867287254963e-03
            3.43570040745255766978516476228833e-03 4.37320177690282488325301812892576e-02 9.52832281823519200436578557855682e-01 5.50056608404268093659439387010934e-04
            9.96564299592547442330214835237712e-01 6.18908320655011404349057912988030e-05 3.37380957538705632606235162995745e-03 1.22852722877149884353335015235720e-06
            9.81985963638956738108731769898441e-01 3.24505506016986773089316731244480e-04 1.76895308550262739255565236362600e-02 1.48478542193902236502557265285240e-05
            9.56117214125662862045373913133517e-01 7.90506100364179922074248096919291e-04 4.30922797739729593335944457521691e-02 5.58315149059668101990880351159774e-05
            9.19558485911109335653179641667521e-01 1.44907635973464126062648649195808e-03 7.89924377291560259051195203028328e-02 1.35992604820674373514166610732445e-04
            8.73165953230075397861753572215093e-01 2.28479313033166930857476017990848e-03 1.24549253639592932829671667604998e-01 2.62451896938059982614110943188734e-04
            8.18026840363257568000676656083670e-01 3.27807111443018713708985600874257e-03 1.78695088522312250933765653826413e-01 4.36632494625845222341320450709645e-04
            7.55433500975413618760967437992804e-01 4.40562980612192300194918104239150e-03 2.40160869218464467778062498837244e-01 6.53817598642139662448724557464175e-04
            6.86853044357709663358946272637695e-01 5.64104064529018494472678568740776e-03 3.07505914997000140420624347825651e-01 9.03320180934016891960425255092559e-04
            6.13892925570822534098169853677973e-01 6.95534687802319695482111683304538e-03 3.79151727551154282824796837303438e-01 1.16925510735767603821833038324485e-03
            5.38263260566748669155856532597682e-01 8.31774241338009701074440727097681e-03 4.53418997019871206077823444502428e-01 1.43184699672974178469297967808416e-03
            4.61736739433251219821841004886664e-01 9.69629394766305732766831226854265e-03 5.28566966619085754075513250427321e-01 1.66915596545002165203286370598335e-03
            3.86107074429177521412981377579854e-01 1.10586894830199521794211747760528e-02 6.02834236087802621817388626368484e-01 1.85906316183307709563599097180031e-03
            3.13146955642290225618751264846651e-01 1.23729957157529685263241958637082e-02 6.74480048641956764221561115846271e-01 1.98133242276520840294273817505655e-03
            2.44566499024586769817091180811985e-01 1.36084065549212200607609446478818e-02 7.41825094420491937263761883514235e-01 2.01955590610926371056277339732787e-03
            1.81973159636742376488172112658503e-01 1.47359652466129628645141735887591e-02 8.03290875116644653708419809845509e-01 1.96280100148675269028375467428305e-03
            1.26834046769924602138246427784907e-01 1.57292432307114815603910074059968e-02 8.57436709999363944056938180438010e-01 1.80680240521432534939183511824012e-03
            8.04415140888906643468203583324794e-02 1.65649600013085089578179776026445e-02 9.02993525909800864859278135554632e-01 1.55458478374509694724947550525940e-03
            4.38827858743368603988699305773480e-02 1.72235302606789757173810784252055e-02 9.38893683864984063269787384342635e-01 1.21645587053592347907826631825401e-03
            1.80140363610431508689657675859053e-02 1.76895308550261663727010130742201e-02 9.64296432783930734800037498644087e-01 8.09390196698491588486734649166010e-04
            3.43570040745255766978516476228833e-03 1.79521455289776490915620854593726e-02 9.78612154063569827933122269314481e-01 3.56348409953123623512133422153170e-04
            9.96564299592547442330214835237712e-01 1.18040372897696700763307472992203e-05 3.42389637016278806577740390082454e-03 5.32968604421346327884329453078438e-07
            9.81985963638956738108731769898441e-01 6.18908320655015199056661612253549e-05 1.79521455289777601138645479750267e-02 6.44140394826517487697785033229358e-06
            9.56117214125662862045373913133517e-01 1.50768105308613436588496825585537e-04 4.37320177690285263882863375783927e-02 2.42212332663709382181080331797318e-05
            9.19558485911109335653179641667521e-01 2.76372942731302299879114769964872e-04 8.01651411461593577101325536204968e-02 5.89972994537343154277178602917786e-05
            8.73165953230075397861753572215093e-01 4.35763786166286728497587521502510e-04 1.26398282983758325492829044378595e-01 1.13858787955956416602584790886965e-04
            8.18026840363257568000676656083670e-01 6.25205258709385066143160436524795e-04 1.81347954378033054956276259872539e-01 1.89423079811144461125430038528350e-04
            7.55433500975413618760967437992804e-01 8.40257220348017029240561548419919e-04 2.43726241804238352717248972112429e-01 2.83643898916977185362120472689185e-04
            6.86853044357709663358946272637695e-01 1.07587912309274483582677284232432e-03 3.12071076519197576626396539722919e-01 3.91884921150239477326965298331629e-04
            6.13892925570822534098169853677973e-01 1.32654823293664012213988190325153e-03 3.84780526196240812986104629089823e-01 5.07254742252733945538645254913490e-04
            5.38263260566748669155856532597682e-01 1.58638910380663701246706231984263e-03 4.60150350329444679520207728273817e-01 6.21174262743089383026484640026865e-04
            4.61736739433251219821841004886664e-01 1.84931130364592109099897143664748e-03 5.36413949263102818321158338221721e-01 7.24125362982025130333119111725182e-04
            3.86107074429177521412981377579854e-01 2.10915217451591733080484836193591e-03 6.11783773396306629344110206147889e-01 8.06512282095783804118649396031060e-04
            3.13146955642290225618751264846651e-01 2.35982128435981326763926091416579e-03 6.84493223073349921214969526772620e-01 8.59556020839606117756037662047675e-04
            2.44566499024586769817091180811985e-01 2.59544318710453912266156173416221e-03 7.52838057788308590012604781804839e-01 8.76138409977513465086429444994565e-04
            1.81973159636742376488172112658503e-01 2.81049514874317238680156982866265e-03 8.15216345214514359618362959736260e-01 8.51516585078301098572783889295579e-04
            1.26834046769924602138246427784907e-01 2.99993662128627104970779448933627e-03 8.70166016608789116837385790859116e-01 7.83840141121788032581296779000013e-04
            8.04415140888906643468203583324794e-02 3.15932746472125536990604999232346e-03 9.16399158446388084620082281617215e-01 6.74421260874951983971259394934350e-04
            4.38827858743368603988699305773480e-02 3.28493230214394480240280849159262e-03 9.52832281823519089414276095340028e-01 5.27731720124757926713987554734331e-04
            1.80140363610431508689657675859053e-02 3.37380957538705632606235162995745e-03 9.78612154063569827933122269314481e-01 3.51135533233630080674619877711962e-04
            3.43570040745255766978516476228833e-03 3.42389637016278806577740390082454e-03 9.93140403222384615666840090852929e-01 1.54593655144624330267238976333033e-04
            ];
        
    case  38
        
        % ALG. DEG.:   38
        % PTS CARD.:  400
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.5e-15
        
        xyw_bar=[
            9.96564299592547442330214835237712e-01 3.42389637016278806577740390082454e-03 1.18040372897696040077608614637938e-05 5.32968604421356704037933318257592e-07
            9.81985963638956738108731769898441e-01 1.79521455289777601138645479750267e-02 6.18908320655017774036821265326580e-05 6.44140394826530023785404396874377e-06
            9.56117214125662862045373913133517e-01 4.37320177690285263882863375783927e-02 1.50768105308611566339749288090388e-04 2.42212332663714091684267065707203e-05
            9.19558485911109335653179641667521e-01 8.01651411461593577101325536204968e-02 2.76372942731306636687804711982608e-04 5.89972994537354606162625481058370e-05
            8.73165953230075397861753572215093e-01 1.26398282983758325492829044378595e-01 4.35763786166276645417383406311274e-04 1.13858787955958639217038386171055e-04
            8.18026840363257568000676656083670e-01 1.81347954378032999445125028614711e-01 6.25205258709432554198315301619004e-04 1.89423079811148147412816489243426e-04
            7.55433500975413618760967437992804e-01 2.43726241804238352717248972112429e-01 8.40257220348028521783589894766919e-04 2.83643898916982714793200148761798e-04
            6.86853044357709663358946272637695e-01 3.12071076519197576626396539722919e-01 1.07587912309276001465718763938639e-03 3.91884921150247120952281321137889e-04
            6.13892925570822534098169853677973e-01 3.84780526196240812986104629089823e-01 1.32654823293665291572551723220386e-03 5.07254742252743811778414873003840e-04
            5.38263260566748669155856532597682e-01 4.60150350329444679520207728273817e-01 1.58638910380665132393573912850115e-03 6.21174262743101526090816477676526e-04
            4.61736739433251219821841004886664e-01 5.36413949263102818321158338221721e-01 1.84931130364596185700065689161420e-03 7.24125362982039224961361423282824e-04
            3.86107074429177521412981377579854e-01 6.11783773396306518321807743632235e-01 2.10915217451601577636211004573852e-03 8.06512282095799525050150435845353e-04
            3.13146955642290225618751264846651e-01 6.84493223073349921214969526772620e-01 2.35982128435985316627920838072896e-03 8.59556020839622922889711187366402e-04
            2.44566499024586769817091180811985e-01 7.52838057788308590012604781804839e-01 2.59544318710458465915280612534843e-03 8.76138409977530595480754715964622e-04
            1.81973159636742376488172112658503e-01 8.15216345214514359618362959736260e-01 2.81049514874320838231369634740986e-03 8.51516585078317686866022917513419e-04
            1.26834046769924602138246427784907e-01 8.70166016608789116837385790859116e-01 2.99993662128628102436778135597706e-03 7.83840141121803319831928824612532e-04
            8.04415140888906643468203583324794e-02 9.16399158446388084620082281617215e-01 3.15932746472125103309736005030572e-03 6.74421260874965102817546469538001e-04
            4.38827858743368603988699305773480e-02 9.52832281823519089414276095340028e-01 3.28493230214399467570274282479659e-03 5.27731720124768226634626167026454e-04
            1.80140363610431508689657675859053e-02 9.78612154063569827933122269314481e-01 3.37380957538702119791196309961379e-03 3.51135533233636911148306536389896e-04
            3.43570040745255766978516476228833e-03 9.93140403222384615666840090852929e-01 3.42389637016282666337474438478239e-03 1.54593655144627366033321935745448e-04
            9.96564299592547442330214835237712e-01 3.37380957538705632606235162995745e-03 6.18908320655013437228131323308844e-05 1.22852722877150498452221774603466e-06
            9.81985963638956738108731769898441e-01 1.76895308550262739255565236362600e-02 3.24505506016987965711706465299358e-04 1.48478542193902998832209794155546e-05
            9.56117214125662862045373913133517e-01 4.30922797739729523947005418449407e-02 7.90506100364185559925545021542348e-04 5.58315149059670948021583125608913e-05
            9.19558485911109335653179641667521e-01 7.89924377291560120273317124883761e-02 1.44907635973465231948864584410330e-03 1.35992604820675078245578726310328e-04
            8.73165953230075397861753572215093e-01 1.24549253639592918951883859790541e-01 2.28479313033168318636256799436524e-03 2.62451896938061337866826550069277e-04
            8.18026840363257568000676656083670e-01 1.78695088522312167667038806939672e-01 3.27807111443026433228453697665827e-03 4.36632494625847444955774045993735e-04
            7.55433500975413618760967437992804e-01 2.40160869218464440022486883208330e-01 4.40562980612194121654567879886599e-03 6.53817598642143023475459262527920e-04
            6.86853044357709663358946272637695e-01 3.07505914997000140420624347825651e-01 5.64104064529019622042937953665387e-03 9.03320180934021554029766942761626e-04
            6.13892925570822534098169853677973e-01 3.79151727551154227313645606045611e-01 6.95534687802323858818454027641565e-03 1.16925510735768210975049630206968e-03
            5.38263260566748669155856532597682e-01 4.53418997019871206077823444502428e-01 8.31774241338012476632002289989032e-03 1.43184699672974915726775257951431e-03
            4.61736739433251219821841004886664e-01 5.28566966619085643053210787911667e-01 9.69629394766313712494820720166899e-03 1.66915596545003010880980909291793e-03
            3.86107074429177521412981377579854e-01 6.02834236087802399772783701337175e-01 1.10586894830201343253861523407977e-02 1.85906316183308663661510884423933e-03
            3.13146955642290225618751264846651e-01 6.74480048641956764221561115846271e-01 1.23729957157530101596876193070784e-02 1.98133242276521837760272504169734e-03
            2.44566499024586769817091180811985e-01 7.41825094420491826241459420998581e-01 1.36084065549213484302981669316068e-02 2.01955590610927411890362925817044e-03
            1.81973159636742376488172112658503e-01 8.03290875116644542686117347329855e-01 1.47359652466130253145593087538145e-02 1.96280100148676266494374154092384e-03
            1.26834046769924602138246427784907e-01 8.57436709999363833034635717922356e-01 1.57292432307115648271178542927373e-02 1.80680240521433467353051849357826e-03
            8.04415140888906643468203583324794e-02 9.02993525909800753836975673038978e-01 1.65649600013085818162039686285425e-02 1.55458478374510497034555189799221e-03
            4.38827858743368603988699305773480e-02 9.38893683864983952247484921826981e-01 1.72235302606791318424939163378440e-02 1.21645587053592976745086673417973e-03
            1.80140363610431508689657675859053e-02 9.64296432783930623777735036128433e-01 1.76895308550262253532991962856613e-02 8.09390196698495708454990094082859e-04
            3.43570040745255766978516476228833e-03 9.78612154063569716910819806798827e-01 1.79521455289777254193950284388848e-02 3.56348409953125466655826647510708e-04
            9.96564299592547442330214835237712e-01 3.28493230214394436872193949739085e-03 1.50768105308613301063225264897483e-04 1.89634498686058429013309451632852e-06
            9.81985963638956738108731769898441e-01 1.72235302606790763313426850800170e-02 7.90506100364185559925545021542348e-04 2.29190312230470312308827640634945e-05
            9.56117214125662862045373913133517e-01 4.19570869782442124051868859169190e-02 1.92569889609292554943920094956411e-03 8.61810881527108867822262183722160e-05
            9.19558485911109335653179641667521e-01 7.69115163507203963844105487623892e-02 3.52999773817026796240980957009015e-03 2.09917117311011358236558232981395e-04
            8.73165953230075397861753572215093e-01 1.21268215453944339099479066135245e-01 5.56583131598026303876736164966132e-03 4.05118687966101993867124031467597e-04
            8.18026840363257568000676656083670e-01 1.73987670437526636968428306317946e-01 7.98548919921579503089503759838408e-03 6.73982491305578088405803249827386e-04
            7.55433500975413618760967437992804e-01 2.33834239715854180774456949620799e-01 1.07322593087322004645756123863976e-02 1.00922771304473601373941082925967e-03
            6.86853044357709663358946272637695e-01 2.99405194840639155007266936081578e-01 1.37417608016511816337867912807269e-02 1.39435794057018135061465358148780e-03
            6.13892925570822534098169853677973e-01 3.69163620357435107166566012892872e-01 1.69434540717423587352641334291548e-02 1.80485300550979848970001295782595e-03
            5.38263260566748669155856532597682e-01 4.41474444966387380429040376839112e-01 2.02622944668639504151030905632069e-02 2.21018778469814431222872919136080e-03
            4.61736739433251219821841004886664e-01 5.14642769159275648149787230067886e-01 2.36204914074731320283717650454491e-02 2.57649604602969671540702023548874e-03
            3.86107074429177521412981377579854e-01 5.86953593768227643856505437724991e-01 2.69393318025948902416644159529824e-02 2.86963530366737872359661487564608e-03
            3.13146955642290225618751264846651e-01 6.56712019285023762549258208309766e-01 3.01410250726860118319905268435832e-02 3.05836917507506036995046372339857e-03
            2.44566499024586769817091180811985e-01 7.22282974409808264937282729079016e-01 3.31505265656049097344748588511720e-02 3.11737064392515013031004222909814e-03
            1.81973159636742376488172112658503e-01 7.82129543688136141810218759928830e-01 3.58972966751214261904578961548395e-02 3.02976421865423914056569820729692e-03
            1.26834046769924602138246427784907e-01 8.34848998671718578457046078256099e-01 3.83169545583568194047074939589947e-02 2.78896600997771953317494464386073e-03
            8.04415140888906643468203583324794e-02 8.79205697774942507294326787814498e-01 4.03527881361668283588528538530227e-02 2.39964486929013925964326858775166e-03
            4.38827858743368603988699305773480e-02 9.14160127147418899440367567876820e-01 4.19570869782441846496112702880055e-02 1.87771173304371753048913618044935e-03
            1.80140363610431508689657675859053e-02 9.38893683864983841225182459311327e-01 4.30922797739730079058517731027678e-02 1.24936835421884571105388683776027e-03
            3.43570040745255766978516476228833e-03 9.52832281823518867369671170308720e-01 4.37320177690285749605436649289913e-02 5.50056608404269177861611872515368e-04
            9.96564299592547442330214835237712e-01 3.15932746472125536990604999232346e-03 2.76372942731302299879114769964872e-04 2.51980644655460147413299915042373e-06
            9.81985963638956738108731769898441e-01 1.65649600013086130412265362110702e-02 1.44907635973464885004169389048911e-03 3.04541225487817237324150027033554e-05
            9.56117214125662862045373913133517e-01 4.03527881361668769311101812036213e-02 3.52999773817026102351590566286177e-03 1.14514849883828889352474500462620e-04
            9.19558485911109335653179641667521e-01 7.39706768999774666450974791587214e-02 6.47083718891319770172287917375797e-03 2.78931581071715812159689251714667e-04
            8.73165953230075397861753572215093e-01 1.16631324009730696911191216713632e-01 1.02027227601939052270552110712742e-02 5.38309584294945921677244538017248e-04
            8.18026840363257568000676656083670e-01 1.67334963152023424859393685437681e-01 1.46381964847190071399296584786498e-02 8.95567757039970749680102279910443e-04
            7.55433500975413618760967437992804e-01 2.24893199547629457635267158366332e-01 1.96732994769569236037654036408640e-02 1.34103157125525936609056198989265e-03
            6.86853044357709663358946272637695e-01 2.87956940398097827049639363394817e-01 2.51900152441925095914143639674876e-02 1.85278108772285714928029776160656e-03
            6.13892925570822534098169853677973e-01 3.55048036761662411109341519477312e-01 3.10590376675150547924886268447153e-02 2.39823464078447708483499134501926e-03
            5.38263260566748669155856532597682e-01 4.24593937002772980893183785155998e-01 3.71428024304783499509596822463209e-02 2.93683136062629183971051105572769e-03
            4.61736739433251219821841004886664e-01 4.94964548908336410271147087769350e-01 4.32987116584123699070119073439855e-02 3.42357081189962296485784776223227e-03
            3.86107074429177521412981377579854e-01 5.64510449149446813521535659674555e-01 4.93824764213757205766341940034181e-02 3.81308548156767193604288479491515e-03
            3.13146955642290225618751264846651e-01 6.31601545513011619625842740788357e-01 5.52514988446981547554059943649918e-02 4.06386939965822872183887071173558e-03
            2.44566499024586769817091180811985e-01 6.94665286363479461684278248867486e-01 6.07682146119337129874793390627019e-02 4.14226878510484848583406147781716e-03
            1.81973159636742376488172112658503e-01 7.52223522759085883038210340600926e-01 6.58033176041716849624663154827431e-02 4.02585998992950350877650933512086e-03
            1.26834046769924602138246427784907e-01 8.02927161901378694253139656211715e-01 7.02387913286967036086139160033781e-02 3.70589453915654140278568284827543e-03
            8.04415140888906643468203583324794e-02 8.45587809011131841252506546879886e-01 7.39706768999774944006730947876349e-02 3.18857626274490985399112652487474e-03
            4.38827858743368603988699305773480e-02 8.79205697774942618316629250330152e-01 7.69115163507204657733495878346730e-02 2.49504713671733430527255315212187e-03
            1.80140363610431508689657675859053e-02 9.02993525909800864859278135554632e-01 7.89924377291559842717560968594626e-02 1.66012326601700107263359651454948e-03
            3.43570040745255766978516476228833e-03 9.16399158446388084620082281617215e-01 8.01651411461593577101325536204968e-02 7.30898753882136412510228673511392e-04
            9.96564299592547442330214835237712e-01 2.99993662128627104970779448933627e-03 4.35763786166286620077370272952066e-04 3.08422457640215244505208121161832e-06
            9.81985963638956738108731769898441e-01 1.57292432307115787049056621071941e-02 2.28479313033168318636256799436524e-03 3.72756222392179873879013152215833e-05
            9.56117214125662862045373913133517e-01 3.83169545583568749158587252168218e-02 5.56583131598026303876736164966132e-03 1.40165334864363760620370458198636e-04
            9.19558485911109335653179641667521e-01 7.02387913286967591197651472612051e-02 1.02027227601939052270552110712742e-02 3.41410205792746287035283048183487e-04
            8.73165953230075397861753572215093e-01 1.10747171349889178548764334664156e-01 1.60868754200354235894820931207505e-02 6.58886975968105180825884481521371e-04
            8.18026840363257568000676656083670e-01 1.58892767396504847043914310233959e-01 2.30803922402375849554090336823720e-02 1.09616835446737039862219464936288e-03
            7.55433500975413618760967437992804e-01 2.13547140248945277463121783512179e-01 3.10193587756411037759107784950174e-02 1.64141279003868971290980294241990e-03
            6.86853044357709663358946272637695e-01 2.73429260024496578740382801697706e-01 3.97176956177937579006709256645991e-02 2.26779043813519478808848894857420e-03
            6.13892925570822534098169853677973e-01 3.37135551692828439040994226161274e-01 4.89715227363490268608359201607527e-02 2.93542157938367440778160144532194e-03
            5.38263260566748669155856532597682e-01 4.03172800228581817982842494529905e-01 5.85639392046695128613009728724137e-02 3.59466000715142957808168056033082e-03
            4.61736739433251219821841004886664e-01 4.69993153001493635390062308943016e-01 6.82701075652551447880966861703200e-02 4.19042551921064172076869525085385e-03
            3.86107074429177521412981377579854e-01 5.36030401537246903309608114795992e-01 7.78625240335756307885617388819810e-02 4.66718861294032096032058376522400e-03
            3.13146955642290225618751264846651e-01 5.99736693205578874632522001775214e-01 8.71163511521308997487267333781347e-02 4.97414628605799720439950561967635e-03
            2.44566499024586769817091180811985e-01 6.59618812981129787331724401155952e-01 9.58146879942833873400331867742352e-02 5.07010655780819850801943715623565e-03
            1.81973159636742376488172112658503e-01 7.14273185833570467551112415094394e-01 1.03753654529687100449564240989275e-01 4.92762304782270923325659950364752e-03
            1.26834046769924602138246427784907e-01 7.62418781880186191557413621922024e-01 1.10747171349889206304339950293070e-01 4.53598768701041210604785192117561e-03
            8.04415140888906643468203583324794e-02 8.02927161901378694253139656211715e-01 1.16631324009730641400039985455805e-01 3.90279391765865901753329580969876e-03
            4.38827858743368603988699305773480e-02 8.34848998671718689479348540771753e-01 1.21268215453944394610630297393072e-01 3.05391936307941157940915033464080e-03
            1.80140363610431508689657675859053e-02 8.57436709999363944056938180438010e-01 1.24549253639592905074096051976085e-01 2.03197867991314123567447857965362e-03
            3.43570040745255766978516476228833e-03 8.70166016608789116837385790859116e-01 1.26398282983758325492829044378595e-01 8.94614704501330818171844772024315e-04
            9.96564299592547442330214835237712e-01 2.81049514874317238680156982866265e-03 6.25205258709385282983594933625682e-04 3.57635683079427176055218456918183e-06
            9.81985963638956738108731769898441e-01 1.47359652466130530701349243827281e-02 3.27807111443020882113330571883125e-03 4.32234822448777751481736875138751e-05
            9.56117214125662862045373913133517e-01 3.58972966751213984348822805259260e-02 7.98548919921573951974380634055706e-03 1.62530723805947527891083148610107e-04
            9.19558485911109335653179641667521e-01 6.58033176041716988402541232971998e-02 1.46381964847189655065662350352795e-02 3.95887099445302943941105144887160e-04
            8.73165953230075397861753572215093e-01 1.03753654529687058816200817545905e-01 2.30803922402375433220456102390017e-02 7.64021840450330677087242836620362e-04
            8.18026840363257568000676656083670e-01 1.48858928808563051315871916813194e-01 3.31142308281793806834514271031367e-02 1.27107773285855879183414707256361e-03
            7.55433500975413618760967437992804e-01 2.00061960455786114199128178370302e-01 4.45045385688002670399043836368946e-02 1.90332373612553873085118105734637e-03
            6.86853044357709663358946272637695e-01 2.56162614693435930490750251919962e-01 5.69843409488544061503034754423425e-02 2.62964891930658750673899248795351e-03
            6.13892925570822534098169853677973e-01 3.15845950137201181284751783095999e-01 7.02611242919762846170783632260282e-02 3.40381018198619754791534042226431e-03
            5.38263260566748669155856532597682e-01 3.77713046038215316535513466078555e-01 8.40236933950360143086300013237633e-02 4.16823955341011267300421749837369e-03
            4.61736739433251219821841004886664e-01 4.40313794325042306976314421262941e-01 9.79494662417064732018445738503942e-02 4.85906799531628068311572832271850e-03
            3.86107074429177521412981377579854e-01 5.02180890226056275693622410472017e-01 1.11712035344766258404547443205956e-01 5.41190547672944482354706607907247e-03
            3.13146955642290225618751264846651e-01 5.61864225669821748532228866679361e-01 1.24988818687888025849019868473988e-01 5.76784264791291387958205305608317e-03
            2.44566499024586769817091180811985e-01 6.17964879907471065223489858908579e-01 1.37468621067942109448267729021609e-01 5.87911475694948958831131946567439e-03
            1.81973159636742376488172112658503e-01 6.69167911554694461173653508012649e-01 1.48858928808563106827023148071021e-01 5.71389595994263241318655843770102e-03
            1.26834046769924602138246427784907e-01 7.14273185833570467551112415094394e-01 1.58892767396504930310641157120699e-01 5.25976956183983428150252592558900e-03
            8.04415140888906643468203583324794e-02 7.52223522759085883038210340600926e-01 1.67334963152023452614969301066594e-01 4.52554064752414605260488755789083e-03
            4.38827858743368603988699305773480e-02 7.82129543688136363854823684960138e-01 1.73987670437526720235155153204687e-01 3.54121598615386853556308288659693e-03
            1.80140363610431508689657675859053e-02 8.03290875116644653708419809845509e-01 1.78695088522312195422614422568586e-01 2.35621001386772388475066009050352e-03
            3.43570040745255766978516476228833e-03 8.15216345214514359618362959736260e-01 1.81347954378033082711851875501452e-01 1.03736330805865271942334260302232e-03
            9.96564299592547442330214835237712e-01 2.59544318710454085738503771096930e-03 8.40257220348016812400127051319032e-04 3.98466455122515066691575927326774e-06
            9.81985963638956738108731769898441e-01 1.36084065549213120011051714186578e-02 4.40562980612194989016305868290146e-03 4.81582475212417252929258948412894e-05
            9.56117214125662862045373913133517e-01 3.31505265656049305511565705728572e-02 1.07322593087322074034695162936259e-02 1.81086632088301114013387249634945e-04
            9.19558485911109335653179641667521e-01 6.07682146119337338041610507843870e-02 1.96732994769569305426593075480923e-02 4.41084982869761172384615033337241e-04
            8.73165953230075397861753572215093e-01 9.58146879942834983623356492898893e-02 3.10193587756411037759107784950174e-02 8.51249158862066724988426624776139e-04
            8.18026840363257568000676656083670e-01 1.37468621067942109448267729021609e-01 4.45045385688003225510556148947217e-02 1.41619492226347214874904167913883e-03
            7.55433500975413618760967437992804e-01 1.84753726579443361410781676568149e-01 5.98127724451430198282508854390471e-02 2.12062357859310558413912239927868e-03
            6.86853044357709663358946272637695e-01 2.36561701020647940918451013203594e-01 7.65852546216423957226027141587110e-02 2.92987230488445755602344711121532e-03
            6.13892925570822534098169853677973e-01 2.91678218987408144347028837728431e-01 9.44288554417693215548013085935963e-02 3.79241848980920023479135316790689e-03
            5.38263260566748669155856532597682e-01 3.48811401599033399367755237108213e-01 1.12925337834217931476388230294106e-01 4.64412170689330622980195428795014e-03
            4.61736739433251219821841004886664e-01 4.06622099376380330415514663400245e-01 1.31641161190368449762644331713091e-01 5.41382107797934528786720775883623e-03
            3.86107074429177521412981377579854e-01 4.63755281988005418902787369006546e-01 1.50137643582817115195382484671427e-01 6.02977525117810633087378491268282e-03
            3.13146955642290225618751264846651e-01 5.18871799954765733353667656047037e-01 1.67981244402944041027581079106312e-01 6.42634927764714359438302793137154e-03
            2.44566499024586769817091180811985e-01 5.70679774395969952038853989506606e-01 1.84753726579443222632903598423582e-01 6.55032516970588743476788806674449e-03
            1.81973159636742376488172112658503e-01 6.17964879907471509312699708971195e-01 2.00061960455786058687976947112475e-01 6.36624357081155226728563789606596e-03
            1.26834046769924602138246427784907e-01 6.59618812981130120398631788702914e-01 2.13547140248945277463121783512179e-01 5.86027018898491267734351950480232e-03
            8.04415140888906643468203583324794e-02 6.94665286363479905773488098930102e-01 2.24893199547629429879691542737419e-01 5.04221537349031512925678910619354e-03
            4.38827858743368603988699305773480e-02 7.22282974409808820048795041657286e-01 2.33834239715854264041183796507539e-01 3.94551172488158147927395091869585e-03
            1.80140363610431508689657675859053e-02 7.41825094420492381352971733576851e-01 2.40160869218464467778062498837244e-01 2.62521525723016443953916621012468e-03
            3.43570040745255766978516476228833e-03 7.52838057788309034101814631867455e-01 2.43726241804238408228400203370256e-01 1.15579764434326631716842825881031e-03
            9.96564299592547442330214835237712e-01 2.35982128435981283395839191996401e-03 1.07587912309274483582677284232432e-03 4.29957615429740088022541807943888e-06
            9.81985963638956738108731769898441e-01 1.23729957157530431194336628664132e-02 5.64104064529021877183456723514610e-03 5.19642368920161275410926626783237e-05
            9.56117214125662862045373913133517e-01 3.01410250726859597902862475393704e-02 1.37417608016511781643398393271127e-02 1.95398070572714584831539030851388e-04
            9.19558485911109335653179641667521e-01 5.52514988446981478165120904577634e-02 2.51900152441925165303082678747160e-02 4.75944323539706590914244443979442e-04
            8.73165953230075397861753572215093e-01 8.71163511521308442375755021203076e-02 3.97176956177937579006709256645991e-02 9.18524141181152153738664889459642e-04
            8.18026840363257568000676656083670e-01 1.24988818687887998093444252845075e-01 5.69843409488544339058790910712560e-02 1.52811807350982948455753174243910e-03
            7.55433500975413618760967437992804e-01 1.67981244402943985516429847848485e-01 7.65852546216423957226027141587110e-02 2.28821835653802421392533084087972e-03
            6.86853044357709663358946272637695e-01 2.15085939814255777680074288582546e-01 9.80610158280345589609794387797592e-02 3.16142273340032104594898854088569e-03
            6.13892925570822534098169853677973e-01 2.65198819519729345195457881345646e-01 1.20908254909448120706372264976380e-01 4.09213671471710636884377976230098e-03
            5.38263260566748669155856532597682e-01 3.17145285171531232659702936871327e-01 1.44591454261720098184440530530992e-01 5.01115079874766070067160228518333e-03
            4.61736739433251219821841004886664e-01 3.69707759186178541721545798282023e-01 1.68555501380570238456613196831313e-01 5.84168019949275404978372705500078e-03
            3.86107074429177521412981377579854e-01 4.21654224837980262652337160034222e-01 1.92238700732842271445832693643752e-01 6.50631378186321794598878653914653e-03
            3.13146955642290225618751264846651e-01 4.71767104543453941190023215312976e-01 2.15085939814255833191225519840373e-01 6.93422940831054691879220541750328e-03
            2.44566499024586769817091180811985e-01 5.18871799954765400286760268500075e-01 2.36561701020647774384997319430113e-01 7.06800322599355178299074253800427e-03
            1.81973159636742376488172112658503e-01 5.61864225669821637509926404163707e-01 2.56162614693435930490750251919962e-01 6.86937349370964501354963616108762e-03
            1.26834046769924602138246427784907e-01 5.99736693205578763610219539259560e-01 2.73429260024496634251534032955533e-01 6.32341258301211909392813836916503e-03
            8.04415140888906643468203583324794e-02 6.31601545513011508603540278272703e-01 2.87956940398097827049639363394817e-01 5.44070616384132940690498969615874e-03
            4.38827858743368603988699305773480e-02 6.56712019285023873571560670825420e-01 2.99405194840639210518418167339405e-01 4.25732904507251062875550218222998e-03
            1.80140363610431508689657675859053e-02 6.74480048641956653199258653330617e-01 3.07505914997000195931775579083478e-01 2.83268836680720427489976742663202e-03
            3.43570040745255766978516476228833e-03 6.84493223073349810192667064256966e-01 3.12071076519197632137547770980746e-01 1.24714136583554588942612806334864e-03
            9.96564299592547442330214835237712e-01 2.10915217451591776448571735613768e-03 1.32654823293663990529944740615065e-03 4.51370990085204018028852690491703e-06
            9.81985963638956738108731769898441e-01 1.10586894830200215683602138483366e-02 6.95534687802324032290801625322274e-03 5.45522354140142936868712597586750e-05
            9.56117214125662862045373913133517e-01 2.69393318025947896277028092981709e-02 1.69434540717423483269232775683122e-02 2.05129569543715026659971845113262e-04
            9.19558485911109335653179641667521e-01 4.93824764213756303710134432094492e-02 3.10590376675150339758069151230302e-02 4.99647995132802206878774864406978e-04
            8.73165953230075397861753572215093e-01 7.78625240335755475218348919952405e-02 4.89715227363490546164115357896662e-02 9.64269816706730476321696166053243e-04
            8.18026840363257568000676656083670e-01 1.11712035344766091871093749432475e-01 7.02611242919763401282295944838552e-02 1.60422363287559665429027333516387e-03
            7.55433500975413618760967437992804e-01 1.50137643582817059684231253413600e-01 9.44288554417693215548013085935963e-02 2.40217953597447872113312961062093e-03
            6.86853044357709663358946272637695e-01 1.92238700732842215934681462385925e-01 1.20908254909448120706372264976380e-01 3.31887250752966028038981605163826e-03
            6.13892925570822534098169853677973e-01 2.37028401504919078446320668263070e-01 1.49078672924258387455509478058957e-01 4.29593926052394044917681625861405e-03
            5.38263260566748669155856532597682e-01 2.83456917814211228900944661290850e-01 1.78279821619040101943198806111468e-01 5.26072342092661574680656499936049e-03
            4.61736739433251219821841004886664e-01 3.30436007756611360708376423644950e-01 2.07827252810137419469782571468386e-01 6.13261605512149289565204668406295e-03
            3.86107074429177521412981377579854e-01 3.76864524065903372385122338528163e-01 2.37028401504919161713047515149810e-01 6.83035068605386454926309269808371e-03
            3.13146955642290225618751264846651e-01 4.21654224837980373674639622549876e-01 2.65198819519729400706609112603473e-01 7.27957798905072041295571949603982e-03
            2.44566499024586769817091180811985e-01 4.63755281988005196858182443975238e-01 2.91678218987407977813575143954949e-01 7.42001420501285473274588611047875e-03
            1.81973159636742376488172112658503e-01 5.02180890226056386715924872987671e-01 3.15845950137201181284751783095999e-01 7.21149202584006436544949991684916e-03
            1.26834046769924602138246427784907e-01 5.36030401537247014331910577311646e-01 3.37135551692828383529842994903447e-01 6.63834037561738827398771789489729e-03
            8.04415140888906643468203583324794e-02 5.64510449149446924543838122190209e-01 3.55048036761662411109341519477312e-01 5.71167212721932805730151727630073e-03
            4.38827858743368603988699305773480e-02 5.86953593768227865901110362756299e-01 3.69163620357435218188868475408526e-01 4.46935873963347045367822119033008e-03
            1.80140363610431508689657675859053e-02 6.02834236087802621817388626368484e-01 3.79151727551154227313645606045611e-01 2.97376603377675014297509825667021e-03
            3.43570040745255766978516476228833e-03 6.11783773396306629344110206147889e-01 3.84780526196240812986104629089823e-01 1.30925331444767018181096762674542e-03
            9.96564299592547442330214835237712e-01 1.84931130364592065731810244244571e-03 1.58638910380663701246706231984263e-03 4.62204647224965192947674502699229e-06
            9.81985963638956738108731769898441e-01 9.69629394766311457354301950317677e-03 8.31774241338014731772521059838255e-03 5.58615801164095599775326339564430e-05
            9.56117214125662862045373913133517e-01 2.36204914074731701922882365352052e-02 2.02622944668639677623378503312779e-02 2.10053021591982274100460226229359e-04
            9.19558485911109335653179641667521e-01 4.32987116584123421514362917150720e-02 3.71428024304783221953840666174074e-02 5.11640380972255641965629191503240e-04
            8.73165953230075397861753572215093e-01 6.82701075652551170325210705414065e-02 5.85639392046694851057253572435002e-02 9.87413901758472717026804410522800e-04
            8.18026840363257568000676656083670e-01 9.79494662417064176906933425925672e-02 8.40236933950360143086300013237633e-02 1.64272767765436195422368292895499e-03
            7.55433500975413618760967437992804e-01 1.31641161190368422007068716084177e-01 1.12925337834217959231963845923019e-01 2.45983585428590943375293242922908e-03
            6.86853044357709663358946272637695e-01 1.68555501380570210701037581202399e-01 1.44591454261720125940016146159905e-01 3.39853098719927445084398875962961e-03
            6.13892925570822534098169853677973e-01 2.07827252810137363958631340210559e-01 1.78279821619040101943198806111468e-01 4.39904897307540499601374861526892e-03
            5.38263260566748669155856532597682e-01 2.48535922890801092011159312278323e-01 2.13200816542450238832984155123995e-01 5.38698956363701120936271493633285e-03
            4.61736739433251219821841004886664e-01 2.89727337675947604900272835948272e-01 2.48535922890801175277886159165064e-01 6.27980907631773676430553265959134e-03
            3.86107074429177521412981377579854e-01 3.30436007756611249686073961129296e-01 2.83456917814211284412095892548678e-01 6.99429050297269584018922117252259e-03
            3.13146955642290225618751264846651e-01 3.69707759186178541721545798282023e-01 3.17145285171531232659702936871327e-01 7.45430001104119104027301290216201e-03
            2.44566499024586769817091180811985e-01 4.06622099376379997348607275853283e-01 3.48811401599033177323150312076905e-01 7.59810693058676159955533435663710e-03
            1.81973159636742376488172112658503e-01 4.40313794325042251465163190005114e-01 3.77713046038215316535513466078555e-01 7.38457987107204311127972573558509e-03
            1.26834046769924602138246427784907e-01 4.69993153001493579878911077685189e-01 4.03172800228581817982842494529905e-01 6.79767162460371936194558628585582e-03
            8.04415140888906643468203583324794e-02 4.94964548908336354759995856511523e-01 4.24593937002772980893183785155998e-01 5.84876179155363520173471769680873e-03
            4.38827858743368603988699305773480e-02 5.14642769159275648149787230067886e-01 4.41474444966387435940191608096939e-01 4.57663080913586414744109731600474e-03
            1.80140363610431508689657675859053e-02 5.28566966619085643053210787911667e-01 4.53418997019871206077823444502428e-01 3.04514138206336305858545898672674e-03
            3.43570040745255766978516476228833e-03 5.36413949263102707298855875706067e-01 4.60150350329444735031358959531644e-01 1.34067757925286851553026146177672e-03
            9.96564299592547442330214835237712e-01 1.58638910380663657878619332564085e-03 1.84931130364592109099897143664748e-03 4.62204647224962821255422190658280e-06
            9.81985963638956738108731769898441e-01 8.31774241338014558300173462157545e-03 9.69629394766311630826649547998386e-03 5.58615801164092685981987784771263e-05
            9.56117214125662862045373913133517e-01 2.02622944668639642928908983776637e-02 2.36204914074731736617351884888194e-02 2.10053021591981162793233428587314e-04
            9.19558485911109335653179641667521e-01 3.71428024304783152564901627101790e-02 4.32987116584123490903301956223004e-02 5.11640380972252931460197977742155e-04
            8.73165953230075397861753572215093e-01 5.85639392046694712279375494290434e-02 6.82701075652551309103088783558633e-02 9.87413901758467512856376480101517e-04
            8.18026840363257568000676656083670e-01 8.40236933950359449196909622514795e-02 9.79494662417064870796323816648510e-02 1.64272767765435328060630304491951e-03
            7.55433500975413618760967437992804e-01 1.12925337834217931476388230294106e-01 1.31641161190368449762644331713091e-01 2.45983585428589642332686260317587e-03
            6.86853044357709663358946272637695e-01 1.44591454261720098184440530530992e-01 1.68555501380570238456613196831313e-01 3.39853098719925623624749100315512e-03
            6.13892925570822534098169853677973e-01 1.78279821619040046432047574853641e-01 2.07827252810137419469782571468386e-01 4.39904897307538157724682292837315e-03
            5.38263260566748669155856532597682e-01 2.13200816542450183321832923866168e-01 2.48535922890801147522310543536150e-01 5.38698956363698258642536131901579e-03
            4.61736739433251219821841004886664e-01 2.48535922890801092011159312278323e-01 2.89727337675947715922575298463926e-01 6.27980907631770380455948910025654e-03
            3.86107074429177521412981377579854e-01 2.83456917814211117878642198775196e-01 3.30436007756611416219527654902777e-01 6.99429050297265854363448767117006e-03
            3.13146955642290225618751264846651e-01 3.17145285171531177148551705613500e-01 3.69707759186178597232697029539850e-01 7.45430001104115200899480342400238e-03
            2.44566499024586769817091180811985e-01 3.48811401599033066300847849561251e-01 4.06622099376380108370909738368937e-01 7.59810693058672170091538689007393e-03
            1.81973159636742376488172112658503e-01 3.77713046038215261024362234820728e-01 4.40313794325042306976314421262941e-01 7.38457987107200494736325424582901e-03
            1.26834046769924602138246427784907e-01 4.03172800228581762471691263272078e-01 4.69993153001493635390062308943016e-01 6.79767162460368293275259077290684e-03
            8.04415140888906643468203583324794e-02 4.24593937002772925382032553898171e-01 4.94964548908336410271147087769350e-01 5.84876179155360484407388810268458e-03
            4.38827858743368603988699305773480e-02 4.41474444966387380429040376839112e-01 5.14642769159275648149787230067886e-01 4.57663080913583986131243364070542e-03
            1.80140363610431508689657675859053e-02 4.53418997019871150566672213244601e-01 5.28566966619085754075513250427321e-01 3.04514138206334701239330620126111e-03
            3.43570040745255766978516476228833e-03 4.60150350329444568497905265758163e-01 5.36413949263102818321158338221721e-01 1.34067757925286135979592305744745e-03
            9.96564299592547442330214835237712e-01 1.32654823293664012213988190325153e-03 2.10915217451591776448571735613768e-03 4.51370990085201646336600378450754e-06
            9.81985963638956738108731769898441e-01 6.95534687802324119026975424162629e-03 1.10586894830200198336367378715295e-02 5.45522354140139955312738262449557e-05
            9.56117214125662862045373913133517e-01 1.69434540717423483269232775683122e-02 2.69393318025947896277028092981709e-02 2.05129569543713915352745047471217e-04
            9.19558485911109335653179641667521e-01 3.10590376675150409147008190302586e-02 4.93824764213756234321195393022208e-02 4.99647995132799496373343650645893e-04
            8.73165953230075397861753572215093e-01 4.89715227363490615553054396968946e-02 7.78625240335755475218348919952405e-02 9.64269816706725272151268235631960e-04
            8.18026840363257568000676656083670e-01 7.02611242919762846170783632260282e-02 1.11712035344766147382244980690302e-01 1.60422363287558798067289345112840e-03
            7.55433500975413618760967437992804e-01 9.44288554417693493103769242225098e-02 1.50137643582817031928655637784686e-01 2.40217953597446571070705978456772e-03
            6.86853044357709663358946272637695e-01 1.20908254909448148461947880605294e-01 1.92238700732842188179105846757011e-01 3.31887250752964249947418728936555e-03
            6.13892925570822534098169853677973e-01 1.49078672924258415211085093687871e-01 2.37028401504919050690745052634156e-01 4.29593926052391703040989057171828e-03
            5.38263260566748669155856532597682e-01 1.78279821619040129698774421740382e-01 2.83456917814211228900944661290850e-01 5.26072342092658799123094937044698e-03
            4.61736739433251219821841004886664e-01 2.07827252810137447225358187097299e-01 3.30436007756611305197225192387123e-01 6.13261605512145993590600312472816e-03
            3.86107074429177521412981377579854e-01 2.37028401504919078446320668263070e-01 3.76864524065903427896273569785990e-01 6.83035068605382725270835919673118e-03
            3.13146955642290225618751264846651e-01 2.65198819519729400706609112603473e-01 4.21654224837980373674639622549876e-01 7.27957798905068051431577202947665e-03
            2.44566499024586769817091180811985e-01 2.91678218987407977813575143954949e-01 4.63755281988005196858182443975238e-01 7.42001420501281483410593864391558e-03
            1.81973159636742376488172112658503e-01 3.15845950137201181284751783095999e-01 5.02180890226056386715924872987671e-01 7.21149202584002533417129043868954e-03
            1.26834046769924602138246427784907e-01 3.37135551692828439040994226161274e-01 5.36030401537247014331910577311646e-01 6.63834037561735271215646037035185e-03
            8.04415140888906643468203583324794e-02 3.55048036761662466620492750735139e-01 5.64510449149446813521535659674555e-01 5.71167212721929683227894969377303e-03
            4.38827858743368603988699305773480e-02 3.69163620357435273700019706666353e-01 5.86953593768227754878807900240645e-01 4.46935873963344616754955751503076e-03
            1.80140363610431508689657675859053e-02 3.79151727551154338335948068561265e-01 6.02834236087802510795086163852829e-01 2.97376603377673409678294547120458e-03
            3.43570040745255766978516476228833e-03 3.84780526196240868497255860347650e-01 6.11783773396306518321807743632235e-01 1.30925331444766302607662922241616e-03
            9.96564299592547442330214835237712e-01 1.07587912309274461898633834522343e-03 2.35982128435981326763926091416579e-03 4.29957615429741189165373238534329e-06
            9.81985963638956738108731769898441e-01 5.64104064529021790447282924674255e-03 1.23729957157530448541571388432203e-02 5.19642368920162630663642233663779e-05
            9.56117214125662862045373913133517e-01 1.37417608016511712254459354198843e-02 3.01410250726859667291801514465988e-02 1.95398070572715099827570961465995e-04
            9.19558485911109335653179641667521e-01 2.51900152441925095914143639674876e-02 5.52514988446981547554059943649918e-02 4.75944323539707783536634178034319e-04
            8.73165953230075397861753572215093e-01 3.97176956177937440228831178501423e-02 8.71163511521308581153633099347644e-02 9.18524141181154538983444357569397e-04
            8.18026840363257568000676656083670e-01 5.69843409488543645169400519989722e-02 1.24988818687888067482383291917358e-01 1.52811807350983338768535269025506e-03
            7.55433500975413618760967437992804e-01 7.65852546216423679670270985297975e-02 1.67981244402944013272005463477399e-01 2.28821835653803028545749675970455e-03
            6.86853044357709663358946272637695e-01 9.80610158280345173276160153363890e-02 2.15085939814255833191225519840373e-01 3.16142273340032885220463043651762e-03
            6.13892925570822534098169853677973e-01 1.20908254909448092950796649347467e-01 2.65198819519729345195457881345646e-01 4.09213671471711677718463562314355e-03
            5.38263260566748669155856532597682e-01 1.44591454261720070428864914902078e-01 3.17145285171531288170854168129154e-01 5.01115079874767371109767211123653e-03
            4.61736739433251219821841004886664e-01 1.68555501380570182945461965573486e-01 3.69707759186178597232697029539850e-01 5.84168019949276879493327285786108e-03
            3.86107074429177521412981377579854e-01 1.92238700732842104912378999870270e-01 4.21654224837980429185790853807703e-01 6.50631378186323442586180831881393e-03
            3.13146955642290225618751264846651e-01 2.15085939814255749924498672953632e-01 4.71767104543454052212325677828630e-01 6.93422940831056513338870317397777e-03
            2.44566499024586769817091180811985e-01 2.36561701020647718873846088172286e-01 5.18871799954765400286760268500075e-01 7.06800322599356999758724029447876e-03
            1.81973159636742376488172112658503e-01 2.56162614693435819468447789404308e-01 5.61864225669821748532228866679361e-01 6.86937349370966236078439592915856e-03
            1.26834046769924602138246427784907e-01 2.73429260024496467718080339182052e-01 5.99736693205578985654824464290868e-01 6.32341258301213470643942216042888e-03
            8.04415140888906643468203583324794e-02 2.87956940398097716027336900879163e-01 6.31601545513011619625842740788357e-01 5.44070616384134328469279751061549e-03
            4.38827858743368603988699305773480e-02 2.99405194840639099496115704823751e-01 6.56712019285023984593863133341074e-01 4.25732904507252190445809603147609e-03
            1.80140363610431508689657675859053e-02 3.07505914997000029398321885309997e-01 6.74480048641956875243863578361925e-01 2.83268836680721121379367133386040e-03
            3.43570040745255766978516476228833e-03 3.12071076519197465604094077207264e-01 6.84493223073350032237271989288274e-01 1.24714136583554892519221102276106e-03
            9.96564299592547442330214835237712e-01 8.40257220348018330283168531025240e-04 2.59544318710453955634243072836398e-03 3.98466455122514897284986476466706e-06
            9.81985963638956738108731769898441e-01 4.40562980612195769641870057853339e-03 1.36084065549213033274877915346224e-02 4.81582475212417049641351607380813e-05
            9.56117214125662862045373913133517e-01 1.07322593087322230159808000848898e-02 3.31505265656049166733687627584004e-02 1.81086632088301032698224313222113e-04
            9.19558485911109335653179641667521e-01 1.96732994769569617676818751306200e-02 6.07682146119336991096915312482452e-02 4.41084982869760955544180536236354e-04
            8.73165953230075397861753572215093e-01 3.10193587756411592870620097528445e-02 9.58146879942834428511844180320622e-02 8.51249158862066399727774879124809e-04
            8.18026840363257568000676656083670e-01 4.45045385688003294899495188019500e-02 1.37468621067942109448267729021609e-01 1.41619492226347149822773818783617e-03
            7.55433500975413618760967437992804e-01 5.98127724451431030949777323257877e-02 1.84753726579443278144054829681409e-01 2.12062357859310471677738441087513e-03
            6.86853044357709663358946272637695e-01 7.65852546216425206226929844888218e-02 2.36561701020647829896148550687940e-01 2.92987230488445625498084012861000e-03
            6.13892925570822534098169853677973e-01 9.44288554417694880882550023670774e-02 2.91678218987407977813575143954949e-01 3.79241848980919850006787719109980e-03
            5.38263260566748669155856532597682e-01 1.12925337834218139643205347510957e-01 3.48811401599033177323150312076905e-01 4.64412170689330449507847831114304e-03
            4.61736739433251219821841004886664e-01 1.31641161190368671807249256744399e-01 4.06622099376380108370909738368937e-01 5.41382107797934268578199379362559e-03
            3.86107074429177521412981377579854e-01 1.50137643582817253973260562815994e-01 4.63755281988005307880484906490892e-01 6.02977525117810372878857094747218e-03
            3.13146955642290225618751264846651e-01 1.67981244402944290827761619766534e-01 5.18871799954765511309062731015729e-01 6.42634927764714099229781396616090e-03
            2.44566499024586769817091180811985e-01 1.84753726579443555699810985970544e-01 5.70679774395969618971946601959644e-01 6.55032516970588483268267410153385e-03
            1.81973159636742376488172112658503e-01 2.00061960455786419510459950288350e-01 6.17964879907471176245792321424233e-01 6.36624357081154966520042393085532e-03
            1.26834046769924602138246427784907e-01 2.13547140248945610530029171059141e-01 6.59618812981129787331724401155952e-01 5.86027018898491007525830553959167e-03
            8.04415140888906643468203583324794e-02 2.24893199547629818457750161542208e-01 6.94665286363479461684278248867486e-01 5.04221537349031252717157514098290e-03
            4.38827858743368603988699305773480e-02 2.33834239715854597108091184054501e-01 7.22282974409808486981887654110324e-01 3.94551172488157974455047494188875e-03
            1.80140363610431508689657675859053e-02 2.40160869218464828600545502013119e-01 7.41825094420492048286064346029889e-01 2.62521525723016313849655922751936e-03
            3.43570040745255766978516476228833e-03 2.43726241804238741295307590917218e-01 7.52838057788308701034907244320493e-01 1.15579764434326588348755926460854e-03
            9.96564299592547442330214835237712e-01 6.25205258709385066143160436524795e-04 2.81049514874317282048243882286442e-03 3.57635683079426625483802741622963e-06
            9.81985963638956738108731769898441e-01 3.27807111443020708640982974202416e-03 1.47359652466130548048584003595352e-02 4.32234822448777073855379071698479e-05
            9.56117214125662862045373913133517e-01 7.98548919921573605029685438694287e-03 3.58972966751213984348822805259260e-02 1.62530723805947283945594339371610e-04
            9.19558485911109335653179641667521e-01 1.46381964847189620371192830816653e-02 6.58033176041716988402541232971998e-02 3.95887099445302347629910277859722e-04
            8.73165953230075397861753572215093e-01 2.30803922402375398525986582853875e-02 1.03753654529687058816200817545905e-01 7.64021840450329592885070351115928e-04
            8.18026840363257568000676656083670e-01 3.31142308281793251723001958453096e-02 1.48858928808563106827023148071021e-01 1.27107773285855684027023659865563e-03
            7.55433500975413618760967437992804e-01 4.45045385688002531621165758224379e-02 2.00061960455786114199128178370302e-01 1.90332373612553591192553259503484e-03
            6.86853044357709663358946272637695e-01 5.69843409488543853336217637206573e-02 2.56162614693435930490750251919962e-01 2.62964891930658360361117154013755e-03
            6.13892925570822534098169853677973e-01 7.02611242919762846170783632260282e-02 3.15845950137201181284751783095999e-01 3.40381018198619234374491249184302e-03
            5.38263260566748669155856532597682e-01 8.40236933950359587974787700659363e-02 3.77713046038215372046664697336382e-01 4.16823955341010573411031359114531e-03
            4.61736739433251219821841004886664e-01 9.79494662417064315684811504070240e-02 4.40313794325042362487465652520768e-01 4.85906799531627287686008642708657e-03
            3.86107074429177521412981377579854e-01 1.11712035344766077993305941618019e-01 5.02180890226056497738227335503325e-01 5.41190547672943614992968619503699e-03
            3.13146955642290225618751264846651e-01 1.24988818687888011971232060659531e-01 5.61864225669821748532228866679361e-01 5.76784264791290520596467317204770e-03
            2.44566499024586769817091180811985e-01 1.37468621067942026181540882134868e-01 6.17964879907471176245792321424233e-01 5.87911475694948091469393958163892e-03
            1.81973159636742376488172112658503e-01 1.48858928808563051315871916813194e-01 6.69167911554694461173653508012649e-01 5.71389595994262373956917855366555e-03
            1.26834046769924602138246427784907e-01 1.58892767396504847043914310233959e-01 7.14273185833570578573414877610048e-01 5.25976956183982560788514604155353e-03
            8.04415140888906643468203583324794e-02 1.67334963152023424859393685437681e-01 7.52223522759085883038210340600926e-01 4.52554064752413911371098365066246e-03
            4.38827858743368603988699305773480e-02 1.73987670437526692479579537575773e-01 7.82129543688136363854823684960138e-01 3.54121598615386289771178596197387e-03
            1.80140363610431508689657675859053e-02 1.78695088522312195422614422568586e-01 8.03290875116644653708419809845509e-01 2.35621001386772041530370813688933e-03
            3.43570040745255766978516476228833e-03 1.81347954378032999445125028614711e-01 8.15216345214514470640665422251914e-01 1.03736330805865098469986662621523e-03
            9.96564299592547442330214835237712e-01 4.35763786166286728497587521502510e-04 2.99993662128627104970779448933627e-03 3.08422457640219776131475931668646e-06
            9.81985963638956738108731769898441e-01 2.28479313033168362004343698856701e-03 1.57292432307115787049056621071941e-02 3.72756222392185294889875579738003e-05
            9.56117214125662862045373913133517e-01 5.56583131598026303876736164966132e-03 3.83169545583568749158587252168218e-02 1.40165334864365820604498180657060e-04
            9.19558485911109335653179641667521e-01 1.02027227601939069617786870480813e-02 7.02387913286967591197651472612051e-02 3.41410205792751328575385105779105e-04
            8.73165953230075397861753572215093e-01 1.60868754200354201200351411671363e-02 1.10747171349889178548764334664156e-01 6.58886975968114830225219602510833e-04
            8.18026840363257568000676656083670e-01 2.30803922402375398525986582853875e-02 1.58892767396504902555065541491786e-01 1.09616835446738644481434743482851e-03
            7.55433500975413618760967437992804e-01 3.10193587756411072453577304486316e-02 2.13547140248945277463121783512179e-01 1.64141279003871378219803212061834e-03
            6.86853044357709663358946272637695e-01 3.97176956177937579006709256645991e-02 2.73429260024496578740382801697706e-01 2.26779043813522818151540150211076e-03
            6.13892925570822534098169853677973e-01 4.89715227363490546164115357896662e-02 3.37135551692828383529842994903447e-01 2.93542157938371734218763187129753e-03
            5.38263260566748669155856532597682e-01 5.85639392046694920446192611507286e-02 4.03172800228581817982842494529905e-01 3.59466000715148248714769785294720e-03
            4.61736739433251219821841004886664e-01 6.82701075652551309103088783558633e-02 4.69993153001493635390062308943016e-01 4.19042551921070330345209242750570e-03
            3.86107074429177521412981377579854e-01 7.78625240335755336440470841807837e-02 5.36030401537247014331910577311646e-01 4.66718861294038948189788484910423e-03
            3.13146955642290225618751264846651e-01 8.71163511521308581153633099347644e-02 5.99736693205578874632522001775214e-01 4.97414628605807093014723463397786e-03
            2.44566499024586769817091180811985e-01 9.58146879942834428511844180320622e-02 6.59618812981129787331724401155952e-01 5.07010655780827310112890415894071e-03
            1.81973159636742376488172112658503e-01 1.03753654529687058816200817545905e-01 7.14273185833570467551112415094394e-01 4.92762304782278209164259052954549e-03
            1.26834046769924602138246427784907e-01 1.10747171349889178548764334664156e-01 7.62418781880186191557413621922024e-01 4.53598768701047889290167702824874e-03
            8.04415140888906643468203583324794e-02 1.16631324009730696911191216713632e-01 8.02927161901378694253139656211715e-01 3.90279391765871626340800304433287e-03
            4.38827858743368603988699305773480e-02 1.21268215453944366855054681764159e-01 8.34848998671718689479348540771753e-01 3.05391936307945624853865673742348e-03
            1.80140363610431508689657675859053e-02 1.24549253639592932829671667604998e-01 8.57436709999363944056938180438010e-01 2.03197867991317115965443917957600e-03
            3.43570040745255766978516476228833e-03 1.26398282983758325492829044378595e-01 8.70166016608789116837385790859116e-01 8.94614704501343937018131846627966e-04
            9.96564299592547442330214835237712e-01 2.76372942731302299879114769964872e-04 3.15932746472125536990604999232346e-03 2.51980644655458326292463318296644e-06
            9.81985963638956738108731769898441e-01 1.44907635973465036792473537019532e-03 1.65649600013086130412265362110702e-02 3.04541225487815068919805056024686e-05
            9.56117214125662862045373913133517e-01 3.52999773817026319192025063387064e-03 4.03527881361668769311101812036213e-02 1.14514849883828062648317980265489e-04
            9.19558485911109335653179641667521e-01 6.47083718891319509963766520854733e-03 7.39706768999774666450974791587214e-02 2.78931581071713806385670153531464e-04
            8.73165953230075397861753572215093e-01 1.02027227601939069617786870480813e-02 1.16631324009730696911191216713632e-01 5.38309584294942018549423590201286e-04
            8.18026840363257568000676656083670e-01 1.46381964847189620371192830816653e-02 1.67334963152023480370544916695508e-01 8.95567757039964352887284615434282e-04
            7.55433500975413618760967437992804e-01 1.96732994769569305426593075480923e-02 2.24893199547629457635267158366332e-01 1.34103157125524982511144411745363e-03
            6.86853044357709663358946272637695e-01 2.51900152441925165303082678747160e-02 2.87956940398097827049639363394817e-01 1.85278108772284392201379343845247e-03
            6.13892925570822534098169853677973e-01 3.10590376675150339758069151230302e-02 3.55048036761662411109341519477312e-01 2.39823464078446017128110057115009e-03
            5.38263260566748669155856532597682e-01 3.71428024304783221953840666174074e-02 4.24593937002773036404335016413825e-01 2.93683136062627102302879933404256e-03
            4.61736739433251219821841004886664e-01 4.32987116584123490903301956223004e-02 4.94964548908336410271147087769350e-01 3.42357081189959867872918408693295e-03
            3.86107074429177521412981377579854e-01 4.93824764213756164932256353949924e-02 5.64510449149446924543838122190209e-01 3.81308548156764461414813816020342e-03
            3.13146955642290225618751264846651e-01 5.52514988446981547554059943649918e-02 6.31601545513011619625842740788357e-01 4.06386939965820009890151709441852e-03
            2.44566499024586769817091180811985e-01 6.07682146119336991096915312482452e-02 6.94665286363479461684278248867486e-01 4.14226878510481899553496987209655e-03
            1.81973159636742376488172112658503e-01 6.58033176041716988402541232971998e-02 7.52223522759085883038210340600926e-01 4.02585998992947488583915571780381e-03
            1.26834046769924602138246427784907e-01 7.02387913286967591197651472612051e-02 8.02927161901378694253139656211715e-01 3.70589453915651451457180520776546e-03
            8.04415140888906643468203583324794e-02 7.39706768999774666450974791587214e-02 8.45587809011131841252506546879886e-01 3.18857626274488686890506983218074e-03
            4.38827858743368603988699305773480e-02 7.69115163507204241399861643913027e-02 8.79205697774942618316629250330152e-01 2.49504713671731652435692438984915e-03
            1.80140363610431508689657675859053e-02 7.89924377291560259051195203028328e-02 9.02993525909800864859278135554632e-01 1.66012326601698914640969917400071e-03
            3.43570040745255766978516476228833e-03 8.01651411461593577101325536204968e-02 9.16399158446388084620082281617215e-01 7.30898753882131208339800743090109e-04
            9.96564299592547442330214835237712e-01 1.50768105308612487911595900769157e-04 3.28493230214394523608367748579440e-03 1.89634498686058047848483187197699e-06
            9.81985963638956738108731769898441e-01 7.90506100364179813654030848368848e-04 1.72235302606790832702365889872453e-02 2.29190312230469871851695068398769e-05
            9.56117214125662862045373913133517e-01 1.92569889609291080428965514670381e-03 4.19570869782442262829746937313757e-02 8.61810881527107105993731894777454e-05
            9.19558485911109335653179641667521e-01 3.52999773817024064051506293537841e-03 7.69115163507204241399861643913027e-02 2.09917117311010924555689238779621e-04
            8.73165953230075397861753572215093e-01 5.56583131598022834429784211351944e-03 1.21268215453944366855054681764159e-01 4.05118687966101180715494667339271e-04
            8.18026840363257568000676656083670e-01 7.98548919921568574331605105953713e-03 1.73987670437526747990730768833600e-01 6.73982491305576787363196267222065e-04
            7.55433500975413618760967437992804e-01 1.07322593087321380145304772213422e-02 2.33834239715854236285608180878626e-01 1.00922771304473406217550035535169e-03
            6.86853044357709663358946272637695e-01 1.37417608016510896934425645099509e-02 2.99405194840639266029569398597232e-01 1.39435794057017853168900511917627e-03
            6.13892925570822534098169853677973e-01 1.69434540717422407740677670062723e-02 3.69163620357435218188868475408526e-01 1.80485300550979480341262650711087e-03
            5.38263260566748669155856532597682e-01 2.02622944668638393928006280475529e-02 4.41474444966387491451342839354766e-01 2.21018778469813997542003924934306e-03
            4.61736739433251219821841004886664e-01 2.36204914074730244755162544834093e-02 5.14642769159275759172089692583540e-01 2.57649604602969151123659230506746e-03
            3.86107074429177521412981377579854e-01 2.69393318025946161553552116174615e-02 5.86953593768227865901110362756299e-01 2.86963530366737265206444895682125e-03
            3.13146955642290225618751264846651e-01 3.01410250726857759095977939978184e-02 6.56712019285023984593863133341074e-01 3.05836917507505429841829780457374e-03
            2.44566499024586769817091180811985e-01 3.31505265656047015676577416343207e-02 7.22282974409808486981887654110324e-01 3.11737064392514405877787631027331e-03
            1.81973159636742376488172112658503e-01 3.58972966751211763902773554946180e-02 7.82129543688136363854823684960138e-01 3.02976421865423306903353228847209e-03
            1.26834046769924602138246427784907e-01 3.83169545583566320545720884638285e-02 8.34848998671718800501651003287407e-01 2.78896600997771389532364771923767e-03
            8.04415140888906643468203583324794e-02 4.03527881361666201920357366361714e-02 8.79205697774942729338931712845806e-01 2.39964486929013448915370965153215e-03
            4.38827858743368603988699305773480e-02 4.19570869782439626050063452566974e-02 9.14160127147419121484972492908128e-01 1.87771173304371362736131523263339e-03
            1.80140363610431508689657675859053e-02 4.30922797739726887167321933702624e-02 9.38893683864984174292089846858289e-01 1.24936835421884310896867287254963e-03
            3.43570040745255766978516476228833e-03 4.37320177690282488325301812892576e-02 9.52832281823519200436578557855682e-01 5.50056608404268093659439387010934e-04
            9.96564299592547442330214835237712e-01 6.18908320655011404349057912988030e-05 3.37380957538705632606235162995745e-03 1.22852722877149884353335015235720e-06
            9.81985963638956738108731769898441e-01 3.24505506016986773089316731244480e-04 1.76895308550262739255565236362600e-02 1.48478542193902236502557265285240e-05
            9.56117214125662862045373913133517e-01 7.90506100364179922074248096919291e-04 4.30922797739729593335944457521691e-02 5.58315149059668101990880351159774e-05
            9.19558485911109335653179641667521e-01 1.44907635973464126062648649195808e-03 7.89924377291560259051195203028328e-02 1.35992604820674373514166610732445e-04
            8.73165953230075397861753572215093e-01 2.28479313033166930857476017990848e-03 1.24549253639592932829671667604998e-01 2.62451896938059982614110943188734e-04
            8.18026840363257568000676656083670e-01 3.27807111443018713708985600874257e-03 1.78695088522312250933765653826413e-01 4.36632494625845222341320450709645e-04
            7.55433500975413618760967437992804e-01 4.40562980612192300194918104239150e-03 2.40160869218464467778062498837244e-01 6.53817598642139662448724557464175e-04
            6.86853044357709663358946272637695e-01 5.64104064529018494472678568740776e-03 3.07505914997000140420624347825651e-01 9.03320180934016891960425255092559e-04
            6.13892925570822534098169853677973e-01 6.95534687802319695482111683304538e-03 3.79151727551154282824796837303438e-01 1.16925510735767603821833038324485e-03
            5.38263260566748669155856532597682e-01 8.31774241338009701074440727097681e-03 4.53418997019871206077823444502428e-01 1.43184699672974178469297967808416e-03
            4.61736739433251219821841004886664e-01 9.69629394766305732766831226854265e-03 5.28566966619085754075513250427321e-01 1.66915596545002165203286370598335e-03
            3.86107074429177521412981377579854e-01 1.10586894830199521794211747760528e-02 6.02834236087802621817388626368484e-01 1.85906316183307709563599097180031e-03
            3.13146955642290225618751264846651e-01 1.23729957157529685263241958637082e-02 6.74480048641956764221561115846271e-01 1.98133242276520840294273817505655e-03
            2.44566499024586769817091180811985e-01 1.36084065549212200607609446478818e-02 7.41825094420491937263761883514235e-01 2.01955590610926371056277339732787e-03
            1.81973159636742376488172112658503e-01 1.47359652466129628645141735887591e-02 8.03290875116644653708419809845509e-01 1.96280100148675269028375467428305e-03
            1.26834046769924602138246427784907e-01 1.57292432307114815603910074059968e-02 8.57436709999363944056938180438010e-01 1.80680240521432534939183511824012e-03
            8.04415140888906643468203583324794e-02 1.65649600013085089578179776026445e-02 9.02993525909800864859278135554632e-01 1.55458478374509694724947550525940e-03
            4.38827858743368603988699305773480e-02 1.72235302606789757173810784252055e-02 9.38893683864984063269787384342635e-01 1.21645587053592347907826631825401e-03
            1.80140363610431508689657675859053e-02 1.76895308550261663727010130742201e-02 9.64296432783930734800037498644087e-01 8.09390196698491588486734649166010e-04
            3.43570040745255766978516476228833e-03 1.79521455289776490915620854593726e-02 9.78612154063569827933122269314481e-01 3.56348409953123623512133422153170e-04
            9.96564299592547442330214835237712e-01 1.18040372897696700763307472992203e-05 3.42389637016278806577740390082454e-03 5.32968604421346327884329453078438e-07
            9.81985963638956738108731769898441e-01 6.18908320655015199056661612253549e-05 1.79521455289777601138645479750267e-02 6.44140394826517487697785033229358e-06
            9.56117214125662862045373913133517e-01 1.50768105308613436588496825585537e-04 4.37320177690285263882863375783927e-02 2.42212332663709382181080331797318e-05
            9.19558485911109335653179641667521e-01 2.76372942731302299879114769964872e-04 8.01651411461593577101325536204968e-02 5.89972994537343154277178602917786e-05
            8.73165953230075397861753572215093e-01 4.35763786166286728497587521502510e-04 1.26398282983758325492829044378595e-01 1.13858787955956416602584790886965e-04
            8.18026840363257568000676656083670e-01 6.25205258709385066143160436524795e-04 1.81347954378033054956276259872539e-01 1.89423079811144461125430038528350e-04
            7.55433500975413618760967437992804e-01 8.40257220348017029240561548419919e-04 2.43726241804238352717248972112429e-01 2.83643898916977185362120472689185e-04
            6.86853044357709663358946272637695e-01 1.07587912309274483582677284232432e-03 3.12071076519197576626396539722919e-01 3.91884921150239477326965298331629e-04
            6.13892925570822534098169853677973e-01 1.32654823293664012213988190325153e-03 3.84780526196240812986104629089823e-01 5.07254742252733945538645254913490e-04
            5.38263260566748669155856532597682e-01 1.58638910380663701246706231984263e-03 4.60150350329444679520207728273817e-01 6.21174262743089383026484640026865e-04
            4.61736739433251219821841004886664e-01 1.84931130364592109099897143664748e-03 5.36413949263102818321158338221721e-01 7.24125362982025130333119111725182e-04
            3.86107074429177521412981377579854e-01 2.10915217451591733080484836193591e-03 6.11783773396306629344110206147889e-01 8.06512282095783804118649396031060e-04
            3.13146955642290225618751264846651e-01 2.35982128435981326763926091416579e-03 6.84493223073349921214969526772620e-01 8.59556020839606117756037662047675e-04
            2.44566499024586769817091180811985e-01 2.59544318710453912266156173416221e-03 7.52838057788308590012604781804839e-01 8.76138409977513465086429444994565e-04
            1.81973159636742376488172112658503e-01 2.81049514874317238680156982866265e-03 8.15216345214514359618362959736260e-01 8.51516585078301098572783889295579e-04
            1.26834046769924602138246427784907e-01 2.99993662128627104970779448933627e-03 8.70166016608789116837385790859116e-01 7.83840141121788032581296779000013e-04
            8.04415140888906643468203583324794e-02 3.15932746472125536990604999232346e-03 9.16399158446388084620082281617215e-01 6.74421260874951983971259394934350e-04
            4.38827858743368603988699305773480e-02 3.28493230214394480240280849159262e-03 9.52832281823519089414276095340028e-01 5.27731720124757926713987554734331e-04
            1.80140363610431508689657675859053e-02 3.37380957538705632606235162995745e-03 9.78612154063569827933122269314481e-01 3.51135533233630080674619877711962e-04
            3.43570040745255766978516476228833e-03 3.42389637016278806577740390082454e-03 9.93140403222384615666840090852929e-01 1.54593655144624330267238976333033e-04
            ];
        
    case  39
        
        % ALG. DEG.:   39
        % PTS CARD.:  441
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.4e-15
        
        xyw_bar=[
            9.96876085310194670618955115060089e-01 3.11415584681614798084892115070943e-03 9.75884298918140019596378920141433e-06 4.00722657624771509374158847197056e-07
            9.83613419283153045391543400910450e-01 1.63353904366299144956808930828629e-02 5.11902802170401127757060066869599e-05 4.84958632450872212176721381959865e-06
            9.60049667075200341415097682329360e-01 3.98255314929132075985052097166772e-02 1.24801431886450986397107953962404e-04 1.82799766260300290308341703005013e-05
            9.26681682291658592731664612074383e-01 7.30892775386205589827071094077837e-02 2.29040169720848285628278517833678e-04 4.46843212796075672874757112040811e-05
            8.84219981737838889301883682492189e-01 1.15418331362325959910108963413222e-01 3.61686899835150788007354094588663e-04 8.66451726689483885546910757469163e-05
            8.33569402098706113690695929108188e-01 1.65910682911676959916391638216737e-01 5.19914989616926392912432675075252e-04 1.45013574154188428638745378407293e-04
            7.75809417943609802570392730558524e-01 2.23490229803788242879392100803670e-01 7.00352252601954550215168637805618e-04 2.18742930253724841187581984236488e-04
            7.12171060103719444356329404399730e-01 2.86929786842787526435216705067432e-01 8.99153053493029208453890532837249e-04 3.04894574126892347557676865577037e-04
            6.44010658401200752898319024097873e-01 3.54877261265164611891975710022962e-01 1.11208033363463520970526587916538e-03 3.98816342578939525120168552163591e-04
            5.72780927080447255228534686466446e-01 4.25884476981894299907338563571102e-01 1.33459593765844486412674996245187e-03 4.94480522267093445913654914392055e-04
            5.00000000000000222044604925031308e-01 4.98438042655097113264872632498736e-01 1.56195734490266469052244246995542e-03 5.84953715455381931774547776825557e-04
            4.27219072919552467215709157244419e-01 5.70991608328300537245070245262468e-01 1.78931875214705105037182875094004e-03 6.62959661495971463128495937411344e-04
            3.55989341598799580168588363449089e-01 6.41998824045029725660072017490165e-01 2.01183435617069417133961906074546e-03 7.21487823798058888410955624692633e-04
            2.87828939896280500132519364342443e-01 7.09946298467407199694889641250484e-01 2.22476163631224466143976314924657e-03 7.54396316624961985747299664240018e-04
            2.24190582056390030896153575667995e-01 7.73385855506406594273016708029900e-01 2.42356243720331931967848504427820e-03 7.56957869696493583500096491434306e-04
            1.66430597901293941820455302149639e-01 8.30965402398517682946987861214438e-01 2.60399970018843074370806789374910e-03 7.26302013140601087845538152976133e-04
            1.15780018262161166209267548765638e-01 8.81457753947868627442119304760126e-01 2.76222778997026185976437773206271e-03 6.61715157286759032319312101577680e-04
            7.33183177083414627794866191834444e-02 9.23786807771574181025187044724589e-01 2.89487452008441170647756734979339e-03 5.64772123934544761914011701264826e-04
            3.99503329247995475625998551549856e-02 9.57050553817281546287176752230152e-01 2.99911325791890615022339261486195e-03 4.39287590093348219270896759525158e-04
            1.63865807168467880750029053160688e-02 9.80540694873564877553917540353723e-01 3.07272440958827885992832307238132e-03 2.91099056550266435412083199096855e-04
            3.12391468980532938104488493991084e-03 9.93761929463378490545721888338448e-01 3.11415584681618007323322672164068e-03 1.27875077873199809522231085168187e-04
            9.96876085310194670618955115060089e-01 3.07272440958829056931178591582920e-03 5.11902802170388117330990240816391e-05 9.24518312904443398556281284023095e-07
            9.83613419283153045391543400910450e-01 1.61180606892572138300057105197993e-02 2.68520027589740778450888569750532e-04 1.11886145734677952697476432941670e-05
            9.60049667075200341415097682329360e-01 3.92956835696624678933019936266646e-02 6.54649355137190691600324043974979e-04 4.21742390370521960858175847253193e-05
            9.26681682291658592731664612074383e-01 7.21168811771902978913217907575017e-02 1.20143653115110937701359716811567e-03 1.03092431976696771744363145817402e-04
            8.84219981737838889301883682492189e-01 1.13882779647510132270404881182913e-01 1.89723861465097842771143632489839e-03 1.99901471336864309516143101141950e-04
            8.33569402098706113690695929108188e-01 1.63703369475031224045125100019504e-01 2.72722842626266226417897087230813e-03 3.34564822762809573747366798102121e-04
            7.75809417943609802570392730558524e-01 2.20516864987566257871876018725743e-01 3.67371706882393955773125071573304e-03 5.04667856907941073628387496086134e-04
            7.12171060103719444356329404399730e-01 2.83112407740025717739484889534651e-01 4.71653215625483790418570606561843e-03 7.03430694326899041274314683391822e-04
            6.44010658401200752898319024097873e-01 3.50155893518353300564172059239354e-01 5.83344808044594653750891666277312e-03 9.20120200802463815079046405287500e-04
            5.72780927080447255228534686466446e-01 4.20218413097379894782790188401123e-01 7.00065982217284998867512513243128e-03 1.14082967237293846432533150903055e-03
            5.00000000000000222044604925031308e-01 4.91806709641576300651166775423917e-01 8.19329035842347730422829954477493e-03 1.34956287559459650778059813802656e-03
            4.27219072919552467215709157244419e-01 5.63395006185773317142206906282809e-01 9.38592089467427115323516773059964e-03 1.52953254852856513137937533031163e-03
            3.55989341598799580168588363449089e-01 6.33457525764799411760463954124134e-01 1.05531326364010080709476824267767e-02 1.66456448854826596570055663448784e-03
            2.87828939896280500132519364342443e-01 7.00501011543127383163209742633626e-01 1.16700485605920611931196617661044e-02 1.74048858140813463186991061348863e-03
            2.24190582056390030896153575667995e-01 7.63096554295586870786394229071448e-01 1.27128636480230428063009640027303e-02 1.74639841125939581333026406895215e-03
            1.66430597901293941820455302149639e-01 8.19910049808121765835267069633119e-01 1.36593522905843478554288594750687e-02 1.67567143776683850908693074188704e-03
            1.15780018262161166209267548765638e-01 8.69730639635642899243350711913081e-01 1.44893421021959900585329705791082e-02 1.52666131848950680643950406079057e-03
            7.33183177083414627794866191834444e-02 9.11496538105962761378009417967405e-01 1.51851441856958313536551941069774e-02 1.30300136830383017118306643311598e-03
            3.99503329247995475625998551549856e-02 9.44317735713490646887180446356069e-01 1.57319313617098055502196984889451e-02 1.01349253391419731086531275110474e-03
            1.63865807168467880750029053160688e-02 9.67495358593895904419923681416549e-01 1.61180606892572519939221820095554e-02 6.71602674640703529475305177953715e-04
            3.12391468980532938104488493991084e-03 9.80540694873564766531615077838069e-01 1.63353904366299040873400372220203e-02 2.95024124561872060600042910749607e-04
            9.96876085310194670618955115060089e-01 2.99911325791893433947987723797723e-03 1.24801431886395041565007701933609e-04 1.42940204325317806934188529649488e-06
            9.83613419283153045391543400910450e-01 1.57319313617098159585605543497877e-02 6.54649355137138649896044739762146e-04 1.72987687850593855123926267935985e-05
            9.60049667075200341415097682329360e-01 3.83543038239972725045312529346120e-02 1.59602910080238608037106473602762e-03 6.52057862032217033093628444362366e-05
            9.26681682291658592731664612074383e-01 7.03892265064069849733741079944593e-02 2.92909120193442229496127993115806e-03 1.59391686302551971845045675202357e-04
            8.84219981737838889301883682492189e-01 1.11154567986548338809349445455155e-01 4.62545027561277188876687205265625e-03 3.09068590194344801347359208421040e-04
            8.33569402098706113690695929108188e-01 1.59781640106263722600488108582795e-01 6.64895779503016370881596230901778e-03 5.17272221201773197037099283335237e-04
            7.75809417943609802570392730558524e-01 2.15234093664632780695811220539326e-01 8.95648839175741673379604890214978e-03 7.80269309714553258895686216334298e-04
            7.12171060103719444356329404399730e-01 2.76330077922032069093205564058735e-01 1.14988619742484865504650315415347e-02 1.08757745273759280288383699542010e-03
            6.44010658401200752898319024097873e-01 3.41767448884246982654389057643129e-01 1.42218927145522644472919182589976e-02 1.42260210177308325796285970454846e-03
            5.72780927080447255228534686466446e-01 4.10151528724592318031483273443882e-01 1.70675441949604267399820400896715e-02 1.76384203745056440844340972518012e-03
            5.00000000000000222044604925031308e-01 4.80024833537599948662943916133372e-01 1.99751664623998292924511588353198e-02 2.08656540919480507079164510741975e-03
            4.27219072919552467215709157244419e-01 5.49898138350608189917068102658959e-01 2.28827887298393983783739713544492e-02 2.36481735361249211305190165433032e-03
            3.55989341598799580168588363449089e-01 6.18282218190953081204952468397096e-01 2.57284402102473386264591681538150e-02 2.57359086115095322955248491325619e-03
            2.87828939896280500132519364342443e-01 6.83719589153168327833043349528452e-01 2.84514709505511165232860548712779e-02 2.69097745258049152530799297267095e-03
            2.24190582056390030896153575667995e-01 7.44815573410567699497164539934602e-01 3.09938445330422140955306531395763e-02 2.70011466787061686981030739218568e-03
            1.66430597901293941820455302149639e-01 8.00268026968936618814609573746566e-01 3.33013751297694948760863553616218e-02 2.59076336675277294971375674492720e-03
            1.15780018262161166209267548765638e-01 8.48895099088651905461233582173008e-01 3.53248826491869838406501003191806e-02 2.36037813155794314046165283116352e-03
            7.33183177083414627794866191834444e-02 8.89660440568793453586238229036098e-01 3.70212417228651391454263830382843e-02 2.01457644723548878751273605303140e-03
            3.99503329247995475625998551549856e-02 9.21695363251203159116187180188717e-01 3.83543038239972933212129646562971e-02 1.56696549822537526015786557564979e-03
            1.63865807168467880750029053160688e-02 9.44317735713490646887180446356069e-01 3.92956835696625095266654170700349e-02 1.03836800416623443142627447599580e-03
            3.12391468980532938104488493991084e-03 9.57050553817281435264874289714498e-01 3.98255314929132353540808253455907e-02 4.56138164676146392381012040218025e-04
            9.96876085310194670618955115060089e-01 2.89487452008442731898885114105724e-03 2.29040169720902062056033798853605e-04 1.90388994200102473633787192319744e-06
            9.83613419283153045391543400910450e-01 1.51851441856957879855682946868001e-02 1.20143653115116662288830440274978e-03 2.30410695537549049002716550216263e-05
            9.60049667075200341415097682329360e-01 3.70212417228651322065324791310559e-02 2.92909120193452637836983853958372e-03 8.68507507027498109252947045000326e-05
            9.26681682291658592731664612074383e-01 6.79427419967601753381103435458499e-02 5.37557571158123193022504437976750e-03 2.12301521340599568931500895629938e-04
            8.84219981737838889301883682492189e-01 1.07291222098938363038733712073736e-01 8.48879616322274765938260543407523e-03 4.11663452586254218802663817911025e-04
            8.33569402098706113690695929108188e-01 1.54228186447977588802160653358442e-01 1.22024114533162975071434175333707e-02 6.88979971639896732887864416738921e-04
            7.75809417943609802570392730558524e-01 2.07753305733961807222343054490921e-01 1.64372763224283902072642149505555e-02 1.03927855555366517072146681499589e-03
            7.12171060103719444356329404399730e-01 2.66725806235310003433625070101698e-01 2.11031336609705522100455254985718e-02 1.44859718312816522847730205114658e-03
            6.44010658401200752898319024097873e-01 3.29888801950675214502695098417462e-01 2.61005396481240325989858774846653e-02 1.89483276998195899164245670220907e-03
            5.72780927080447255228534686466446e-01 3.95896089200173828537288045481546e-01 3.13229837193789162341772680520080e-02 2.34934679870603879345281050916583e-03
            5.00000000000000222044604925031308e-01 4.63340841145829074321227381005883e-01 3.66591588541707036341676939628087e-02 2.77919771742596386637780270234543e-03
            4.27219072919552467215709157244419e-01 5.30785593091484875216679029108491e-01 4.19953339889627130787630449049175e-02 3.14981498414917309228155950506789e-03
            3.55989341598799580168588363449089e-01 5.96792880340983100673213357367786e-01 4.72177780602173191581982791831251e-02 3.42789054940730118087310884789076e-03
            2.87828939896280500132519364342443e-01 6.59955876056348644809190773230512e-01 5.22151840473707995471386311692186e-02 3.58424344662286540469375850648248e-03
            2.24190582056390030896153575667995e-01 7.18928376557696924287199635728030e-01 5.68810413859129893054955573461484e-02 3.59641374704385528848771258481065e-03
            1.66430597901293941820455302149639e-01 7.72453495843680948418352727458114e-01 6.11159062550251652723432016500738e-02 3.45076344290047966681389546295122e-03
            1.15780018262161166209267548765638e-01 8.19390460192720215815143092186190e-01 6.48295215451186734867405903059989e-02 3.14390216888501584924231480044909e-03
            7.33183177083414627794866191834444e-02 8.58738940294898500660281115415273e-01 6.79427419967600920713834966591094e-02 2.68331212578548812530687683874930e-03
            3.99503329247995475625998551549856e-02 8.89660440568793564608540691551752e-01 7.03892265064068878288594532932620e-02 2.08711738283523908291106074841537e-03
            1.63865807168467880750029053160688e-02 9.11496538105962872400311880483059e-01 7.21168811771902840135339829430450e-02 1.38305273072679749644942059916275e-03
            3.12391468980532938104488493991084e-03 9.23786807771574181025187044724589e-01 7.30892775386204895937680703354999e-02 6.07552555272165080609003684486424e-04
            9.96876085310194670618955115060089e-01 2.76222778997023497155050009155275e-03 3.61686899835094409494384848358095e-04 2.33781382801618982733441245991379e-06
            9.83613419283153045391543400910450e-01 1.44893421021960368960668219528998e-02 1.89723861465091771238977713665008e-03 2.82924605181941991275274689288111e-05
            9.60049667075200341415097682329360e-01 3.53248826491868866961354456179834e-02 4.62545027561277188876687205265625e-03 1.06645285259018425777326199987982e-04
            9.26681682291658592731664612074383e-01 6.48295215451187567534674371927395e-02 8.48879616322265051486795073287794e-03 2.60688089867886266769170244828047e-04
            8.84219981737838889301883682492189e-01 1.02375005633374704938631793993409e-01 1.34050126287864057594845235144021e-02 5.05487471052837897093901187162146e-04
            8.33569402098706113690695929108188e-01 1.47161260236899682496414243360050e-01 1.92693376643942038128898275317624e-02 8.46008411196851614968772459235424e-04
            7.75809417943609802570392730558524e-01 1.98233792371696804224256993620656e-01 2.59567896846933932053502758208197e-02 1.27614507789271269290465316004202e-03
            7.12171060103719444356329404399730e-01 2.54504099978710762730571559586679e-01 3.33248399175697929130990360135911e-02 1.77875330460698579011780129377485e-03
            6.44010658401200752898319024097873e-01 3.14772889127355570693111985747237e-01 4.12164524714436764085689901548903e-02 2.32669239629801002933184861376503e-03
            5.72780927080447255228534686466446e-01 3.77755640854983365528596550575458e-01 4.94634320645693792428687629580963e-02 2.88479670576335684578639373398801e-03
            5.00000000000000222044604925031308e-01 4.42109990868919222606336916214786e-01 5.78900091310805553490581587539054e-02 3.41261682792479018391795797526811e-03
            4.27219072919552467215709157244419e-01 5.06464340882855745817892056948040e-01 6.63165861975918424775500170653686e-02 3.86770309732145656234081343427533e-03
            3.55989341598799580168588363449089e-01 5.69447092610482985541864309197990e-01 7.45635657907174342895473273529205e-02 4.20915608121117355844287288846317e-03
            2.87828939896280500132519364342443e-01 6.29715881759128182082463354163337e-01 8.24551783445912622738660502363928e-02 4.40114405125988789047042359925399e-03
            2.24190582056390030896153575667995e-01 6.85986189366142196099929151387187e-01 8.98232285774677174927660416869912e-02 4.41608813809370864822412272587826e-03
            1.66430597901293941820455302149639e-01 7.37058721500939206805469439132139e-01 9.65106805977669068852264899760485e-02 4.23724203592707165250130785238980e-03
            1.15780018262161166209267548765638e-01 7.81844976104464128852100657240953e-01 1.02375005633374760449783025251236e-01 3.86044266646245245316881877783999e-03
            7.33183177083414627794866191834444e-02 8.19390460192720215815143092186190e-01 1.07291222098938376916521519888192e-01 3.29487753160337621186060097500103e-03
            3.99503329247995475625998551549856e-02 8.48895099088652016483536044688663e-01 1.11154567986548435953864100156352e-01 2.56280143649320005933001098696877e-03
            1.63865807168467880750029053160688e-02 8.69730639635642899243350711913081e-01 1.13882779647510257170495151513023e-01 1.69827032930820460802923133059039e-03
            3.12391468980532938104488493991084e-03 8.81457753947868627442119304760126e-01 1.15418331362326043176835810299963e-01 7.46022516127706702103450364660375e-04
            9.96876085310194670618955115060089e-01 2.60399970018839344715333439239657e-03 5.19914989616935933891550547514271e-04 2.72191556260654341119274539362305e-06
            9.83613419283153045391543400910450e-01 1.36593522905843027526184840780843e-02 2.72722842626265185583811501146556e-03 3.29408987431015738505356293952531e-05
            9.60049667075200341415097682329360e-01 3.33013751297694601816168358254799e-02 6.64895779503019840328548184515967e-03 1.24167056480909066733600565335394e-04
            9.26681682291658592731664612074383e-01 6.11159062550251722112371055573021e-02 1.22024114533162350570982823683153e-02 3.03519022898284809245522675169582e-04
            8.84219981737838889301883682492189e-01 9.65106805977668652518630665326782e-02 1.92693376643942454462532509751327e-02 5.88538829599144584087122566984362e-04
            8.33569402098706113690695929108188e-01 1.38731453983511721084198597964132e-01 2.76991439177821652251054729276802e-02 9.85007203283938927509955085781712e-04
            7.75809417943609802570392730558524e-01 1.86878409440906084748590387789591e-01 3.73121726154841126810168816518853e-02 1.48581512609474546138255135474537e-03
            7.12171060103719444356329404399730e-01 2.39925397336047036001005494654237e-01 4.79035425602335196426651009460329e-02 2.07100165283736342702880506294605e-03
            6.44010658401200752898319024097873e-01 2.96741822630023133200438678613864e-01 5.92475189687761139012422972882632e-02 2.70896688478224154789097966045119e-03
            5.72780927080447255228534686466446e-01 3.56116747178715054023001584937447e-01 7.11023257408376907484637285961071e-02 3.35876747509724835061284409221116e-03
            5.00000000000000222044604925031308e-01 4.16784701049352890311894270780613e-01 8.32152989506468876435008041880792e-02 3.97330820009040349555240467793737e-03
            4.27219072919552467215709157244419e-01 4.77452654919991170689996806686395e-01 9.53282721604564176054452673270134e-02 4.50316493382804054063095477999923e-03
            3.55989341598799580168588363449089e-01 5.36827579468682647423349862947362e-01 1.07183078932517772408061773603549e-01 4.90071848561643604036408561341887e-03
            2.87828939896280500132519364342443e-01 5.93644004762659105445266050082864e-01 1.18527055341060338911063354316866e-01 5.12424999066886290716160701208537e-03
            2.24190582056390030896153575667995e-01 6.46690992657800167719983619463164e-01 1.29118425285809745872711573611014e-01 5.14164938408268219760532957707255e-03
            1.66430597901293941820455302149639e-01 6.94837948115194392606497331144055e-01 1.38731453983511721084198597964132e-01 4.93341894974908944798697163491852e-03
            1.15780018262161166209267548765638e-01 7.37058721500939206805469439132139e-01 1.47161260236899682496414243360050e-01 4.49471161752478236967034419535594e-03
            7.33183177083414627794866191834444e-02 7.72453495843680948418352727458114e-01 1.54228186447977644313311884616269e-01 3.83622439164208557313417990997095e-03
            3.99503329247995475625998551549856e-02 8.00268026968936729836912036262220e-01 1.59781640106263722600488108582795e-01 2.98386853147355850246480457599318e-03
            1.63865807168467880750029053160688e-02 8.19910049808121876857569532148773e-01 1.63703369475031279556276331277331e-01 1.97729536178657986852935657395847e-03
            3.12391468980532938104488493991084e-03 8.30965402398517682946987861214438e-01 1.65910682911676987671967253845651e-01 8.68593671732201515894855603505675e-04
            9.96876085310194670618955115060089e-01 2.42356243720336485616972943546443e-03 7.00352252601964524875155504446411e-04 3.04800606181929615792628243675644e-06
            9.83613419283153045391543400910450e-01 1.27128636480230167854488243506239e-02 3.67371706882393782300777473892595e-03 3.68872791022954957273517762050830e-05
            9.60049667075200341415097682329360e-01 3.09938445330422106260837011859621e-02 8.95648839175744795881861648467748e-03 1.39042498610665723546633842921949e-04
            9.26681682291658592731664612074383e-01 5.68810413859130517555406925112038e-02 1.64372763224283555127946954144136e-02 3.39881161039957784258253026266061e-04
            8.84219981737838889301883682492189e-01 8.98232285774676758594026182436210e-02 2.59567896846934348387136992641899e-02 6.59046865699387221688987370527002e-04
            8.33569402098706113690695929108188e-01 1.29118425285809773628287189239927e-01 3.73121726154841126810168816518853e-02 1.10301288099843414723666423782333e-03
            7.75809417943609802570392730558524e-01 1.73929164973607164323965434959973e-01 5.02614170827830331056418344815029e-02 1.66381851564226007807179019692967e-03
            7.12171060103719444356329404399730e-01 2.23300402328259706141722062966437e-01 6.45285375680208495019485326338327e-02 2.31911146642668070105841593431251e-03
            6.44010658401200752898319024097873e-01 2.76179883899893341414610858919332e-01 7.98094576989059056870701169827953e-02 3.03350610853521211740524421429654e-03
            5.72780927080447255228534686466446e-01 3.31440580296126774406673121120548e-01 9.57784926234259703647921924130060e-02 3.76115400675188296664153142501164e-03
            5.00000000000000222044604925031308e-01 3.87904708971804734751742671505781e-01 1.12095291028195043203652403462911e-01 4.44931784281894695393111405223863e-03
            4.27219072919552467215709157244419e-01 4.44368837647483194697173303211457e-01 1.28412089432964393598268770801951e-01 5.04265238945768995426632841372339e-03
            3.55989341598799580168588363449089e-01 4.99629534043716239111176946607884e-01 1.44381124357484180720234689943027e-01 5.48783358919646430812422366329884e-03
            2.87828939896280500132519364342443e-01 5.52509015615350151939821898849914e-01 1.59662044488369292416507505549816e-01 5.73814458038494207281488002081460e-03
            2.24190582056390030896153575667995e-01 6.01880252970002693757578526856378e-01 1.73929164973607219835116666217800e-01 5.75762845318609269712473164304356e-03
            1.66430597901293941820455302149639e-01 6.46690992657800056697681156947510e-01 1.86878409440906056993014772160677e-01 5.52445162917900717269859001135046e-03
            1.15780018262161166209267548765638e-01 6.85986189366142085077626688871533e-01 1.98233792371696804224256993620656e