
function xyw=set_lether_pre(degV)
%--------------------------------------------------------------------
% SOURCE:
% F.G. Lether
% Computation of double integrals over a triangle
% JCAM, vol 2, no 3, (1976), pp. 219-224
%--------------------------------------------------------------------
% REMARK.
% Set valid for any degree degV.
%--------------------------------------------------------------------
% | DEG | CARD | W- | PT.O|   M.E.   |Q.|
%--------------------------------------------------------------------
% |   1 |   4 |   0 |   0 | 4.44e-16 |PI|
% |   2 |   4 |   0 |   0 | 4.44e-16 |PI|
% |   3 |   9 |   0 |   0 | 3.89e-16 |PI|
% |   4 |   9 |   0 |   0 | 3.89e-16 |PI|
% |   5 |  16 |   0 |   0 | 5.20e-16 |PI|
% |   6 |  16 |   0 |   0 | 5.20e-16 |PI|
% |   7 |  25 |   0 |   0 | 5.97e-16 |PI|
% |   8 |  25 |   0 |   0 | 5.97e-16 |PI|
% |   9 |  36 |   0 |   0 | 7.29e-16 |PI|
% |  10 |  36 |   0 |   0 | 7.29e-16 |PI|
% |  11 |  49 |   0 |   0 | 1.44e-15 |PI|
% |  12 |  49 |   0 |   0 | 1.44e-15 |PI|
% |  13 |  64 |   0 |   0 | 9.99e-16 |PI|
% |  14 |  64 |   0 |   0 | 9.99e-16 |PI|
% |  15 |  81 |   0 |   0 | 1.55e-15 |PI|
% |  16 |  81 |   0 |   0 | 1.55e-15 |PI|
% |  17 | 100 |   0 |   0 | 9.37e-16 |PI|
% |  18 | 100 |   0 |   0 | 9.37e-16 |PI|
% |  19 | 121 |   0 |   0 | 1.12e-15 |PI|
% |  20 | 121 |   0 |   0 | 1.12e-15 |PI|
% |  21 | 144 |   0 |   0 | 1.32e-15 |PI|
% |  22 | 144 |   0 |   0 | 1.32e-15 |PI|
% |  23 | 169 |   0 |   0 | 1.11e-15 |PI|
% |  24 | 169 |   0 |   0 | 1.11e-15 |PI|
% |  25 | 196 |   0 |   0 | 1.39e-15 |PI|
% |  26 | 196 |   0 |   0 | 1.39e-15 |PI|
% |  27 | 225 |   0 |   0 | 1.33e-15 |PI|
% |  28 | 225 |   0 |   0 | 1.33e-15 |PI|
% |  29 | 256 |   0 |   0 | 1.08e-15 |PI|
% |  30 | 256 |   0 |   0 | 1.66e-15 |PI|
% |  31 | 289 |   0 |   0 | 1.05e-15 |PI|
% |  32 | 289 |   0 |   0 | 1.05e-15 |PI|
% |  33 | 324 |   0 |   0 | 1.20e-15 |PI|
% |  34 | 324 |   0 |   0 | 1.22e-15 |PI|
% |  35 | 361 |   0 |   0 | 1.77e-15 |PI|
% |  36 | 361 |   0 |   0 | 1.77e-15 |PI|
% |  37 | 400 |   0 |   0 | 1.46e-15 |PI|
% |  38 | 400 |   0 |   0 | 1.46e-15 |PI|
% |  39 | 441 |   0 |   0 | 1.43e-15 |PI|
% |  40 | 441 |   0 |   0 | 1.43e-15 |PI|
% |  41 | 484 |   0 |   0 | 1.88e-15 |PI|
% |  42 | 484 |   0 |   0 | 1.88e-15 |PI|
% |  43 | 529 |   0 |   0 | 1.16e-15 |PI|
% |  44 | 529 |   0 |   0 | 1.16e-15 |PI|
% |  45 | 576 |   0 |   0 | 1.61e-15 |PI|
% |  46 | 576 |   0 |   0 | 1.61e-15 |PI|
% |  47 | 625 |   0 |   0 | 1.49e-15 |PI|
% |  48 | 625 |   0 |   0 | 1.49e-15 |PI|
% |  49 | 676 |   0 |   0 | 1.19e-15 |PI|
% |  50 | 676 |   0 |   0 | 1.19e-15 |PI|
%--------------------------------------------------------------------
% LEGEND:
%--------------------------------------------------------------------
% DEG  : algebraic degree of precision
% CARD : cardinality of the pointset
% W-   : negative weights
% PT.O : points outside the simplex [0 0; 1 0; 0 1].
% M.E. : maximum moment error in inf norm.
% Q.: quality of the rule.
%        PI: pos. weights, points in simplex
%        NI: some neg. weights, points in simplex
%        P0: pos. weights, some points out of the simplex
%        N0: some neg. weights, some points out of the simplex
%--------------------------------------------------------------------

deg=ceil((degV+2)/2);

[a,A]=gauss_jacobi(deg,0,0);

A1=(1+a).*A;
[W1,W2]=meshgrid(A1,A);
W=(W1.*W2)'/8;

aa=(1-a)/2;
L=length(aa);
alpha1=repmat(aa,1,L);

[B1,B2]=meshgrid(1+a,(1-a)/4);
beta=B1.*B2; 
beta1=beta';


xyw=[alpha1(:) beta1(:) W(:)];






function [x,w]=gauss_jacobi(N,alpha,beta)
ab=r_jacobi(N, alpha,beta);
xw=gauss(N,ab);
x=xw(:,1); 
w=xw(:,2); 

% R_JACOBI Recurrence coefficients for monic Jacobi polynomials.
%
%    ab=R_JACOBI(n,a,b) generates the first n recurrence 
%    coefficients for monic Jacobi polynomials with parameters 
%    a and b. These are orthogonal on [-1,1] relative to the
%    weight function w(t)=(1-t)^a(1+t)^b. The n alpha-coefficients
%    are stored in the first column, the n beta-coefficients in
%    the second column, of the nx2 array ab. The call ab=
%    R_JACOBI(n,a) is the same as ab=R_JACOBI(n,a,a) and
%    ab=R_JACOBI(n) the same as ab=R_JACOBI(n,0,0).
%
%    Supplied by Dirk Laurie, 6-22-1998; edited by Walter
%    Gautschi, 4-4-2002.
%
function ab=r_jacobi(N,a,b)
if nargin<2, a=0; end;  if nargin<3, b=a; end
if((N<=0)|(a<=-1)|(b<=-1)) error('parameter(s) out of range'), end
nu=(b-a)/(a+b+2);
mu=2^(a+b+1)*gamma(a+1)*gamma(b+1)/gamma(a+b+2);
if N==1, ab=[nu mu]; return, end 
N=N-1; n=1:N; nab=2*n+a+b;
A=[nu (b^2-a^2)*ones(1,N)./(nab.*(nab+2))];
n=2:N; nab=nab(n);
B1=4*(a+1)*(b+1)/((a+b+2)^2*(a+b+3));
B=4*(n+a).*(n+b).*n.*(n+a+b)./((nab.^2).*(nab+1).*(nab-1));
ab=[A' [mu; B1; B']];


% GAUSS Gauss quadrature rule.
%
%    Given a weight function w encoded by the nx2 array ab of the 
%    first n recurrence coefficients for the associated orthogonal
%    polynomials, the first column of ab containing the n alpha-
%    coefficients and the second column the n beta-coefficients, 
%    the call xw=GAUSS(n,ab) generates the nodes and weights xw of
%    the n-point Gauss quadrature rule for the weight function w.
%    The nodes, in increasing order, are stored in the first 
%    column, the n corresponding weights in the second column, of
%    the nx2 array xw.
%
function xw=gauss(N,ab)
N0=size(ab,1); if N0<N, error('input array ab too short'), end
J=zeros(N);
for n=1:N, J(n,n)=ab(n,1); end
for n=2:N
  J(n,n-1)=sqrt(ab(n,2));
  J(n-1,n)=J(n,n-1);
end
[V,D]=eig(J);
[D,I]=sort(diag(D));
V=V(:,I);
xw=[D ab(1,2)*V(1,:)'.^2];

