
function [xyw,pointset_stats]=set_lether_standard(deg)

%--------------------------------------------------------------------
% CUBATURE RULE ON THE SIMPLEX [0 0; 0 1; 1 0] WITH ADE EQUAL TO deg.
% THE CUBATURE RULE IS DESCRIBED IN STANDARD FORM.
%--------------------------------------------------------------------
% INPUT:
% deg: algebraic degree of precision (ADE) of the rule.
%--------------------------------------------------------------------
% OUTPUT:
% xyw: MATRIX OF NODES xyw(:,1:2) AND WEIGHTS xyw(:,3).
%      REMARK: The sum of the weights makes 1/2.
%
% pointset_stats:
%    statistics of the pointset stored in a matrix. Each rows contains
%    degree, cardinality of the full pointset, moment error w.r.t. orthn.
%    Dubiner basis,number of points with negative weights, number of
%    points outside the domain.
%--------------------------------------------------------------------
% SOURCE:
% F.G. Lether,"
% "Computation of double integrals over a triangle",
% JCAM, vol 2, no 3, (1976), pp. 219-224
%--------------------------------------------------------------------
% | DEG | CARD | W- | PT.O|   M.E.   |Q.|
%--------------------------------------------------------------------
% |   1 |   4 |   0 |   0 | 4.44e-16 |PI|
% |   2 |   4 |   0 |   0 | 4.44e-16 |PI|
% |   3 |   9 |   0 |   0 | 3.89e-16 |PI|
% |   4 |   9 |   0 |   0 | 3.89e-16 |PI|
% |   5 |  16 |   0 |   0 | 5.20e-16 |PI|
% |   6 |  16 |   0 |   0 | 5.20e-16 |PI|
% |   7 |  25 |   0 |   0 | 5.97e-16 |PI|
% |   8 |  25 |   0 |   0 | 5.97e-16 |PI|
% |   9 |  36 |   0 |   0 | 7.29e-16 |PI|
% |  10 |  36 |   0 |   0 | 7.29e-16 |PI|
% |  11 |  49 |   0 |   0 | 1.44e-15 |PI|
% |  12 |  49 |   0 |   0 | 1.44e-15 |PI|
% |  13 |  64 |   0 |   0 | 9.99e-16 |PI|
% |  14 |  64 |   0 |   0 | 9.99e-16 |PI|
% |  15 |  81 |   0 |   0 | 1.55e-15 |PI|
% |  16 |  81 |   0 |   0 | 1.55e-15 |PI|
% |  17 | 100 |   0 |   0 | 9.37e-16 |PI|
% |  18 | 100 |   0 |   0 | 9.37e-16 |PI|
% |  19 | 121 |   0 |   0 | 1.12e-15 |PI|
% |  20 | 121 |   0 |   0 | 1.12e-15 |PI|
% |  21 | 144 |   0 |   0 | 1.32e-15 |PI|
% |  22 | 144 |   0 |   0 | 1.32e-15 |PI|
% |  23 | 169 |   0 |   0 | 1.11e-15 |PI|
% |  24 | 169 |   0 |   0 | 1.11e-15 |PI|
% |  25 | 196 |   0 |   0 | 1.39e-15 |PI|
% |  26 | 196 |   0 |   0 | 1.39e-15 |PI|
% |  27 | 225 |   0 |   0 | 1.33e-15 |PI|
% |  28 | 225 |   0 |   0 | 1.33e-15 |PI|
% |  29 | 256 |   0 |   0 | 1.08e-15 |PI|
% |  30 | 256 |   0 |   0 | 1.66e-15 |PI|
% |  31 | 289 |   0 |   0 | 1.05e-15 |PI|
% |  32 | 289 |   0 |   0 | 1.05e-15 |PI|
% |  33 | 324 |   0 |   0 | 1.20e-15 |PI|
% |  34 | 324 |   0 |   0 | 1.22e-15 |PI|
% |  35 | 361 |   0 |   0 | 1.77e-15 |PI|
% |  36 | 361 |   0 |   0 | 1.77e-15 |PI|
% |  37 | 400 |   0 |   0 | 1.46e-15 |PI|
% |  38 | 400 |   0 |   0 | 1.46e-15 |PI|
% |  39 | 441 |   0 |   0 | 1.43e-15 |PI|
% |  40 | 441 |   0 |   0 | 1.43e-15 |PI|
% |  41 | 484 |   0 |   0 | 1.88e-15 |PI|
% |  42 | 484 |   0 |   0 | 1.88e-15 |PI|
% |  43 | 529 |   0 |   0 | 1.16e-15 |PI|
% |  44 | 529 |   0 |   0 | 1.16e-15 |PI|
% |  45 | 576 |   0 |   0 | 1.61e-15 |PI|
% |  46 | 576 |   0 |   0 | 1.61e-15 |PI|
% |  47 | 625 |   0 |   0 | 1.49e-15 |PI|
% |  48 | 625 |   0 |   0 | 1.49e-15 |PI|
% |  49 | 676 |   0 |   0 | 1.19e-15 |PI|
% |  50 | 676 |   0 |   0 | 1.19e-15 |PI|
%--------------------------------------------------------------------
% LEGEND:
%--------------------------------------------------------------------
% DEG  : algebraic degree of precision
% CARD : cardinality of the pointset
% W-   : negative weights
% PT.O : points outside the simplex [0 0; 1 0; 0 1].
% M.E. : maximum moment error in inf norm.
% Q.: quality of the rule.
%        PI: pos. weights, points in simplex
%        NI: some neg. weights, points in simplex
%        P0: pos. weights, some points out of the simplex
%        N0: some neg. weights, some points out of the simplex
%--------------------------------------------------------------------


switch deg
    
    case   1
        
        % ALG. DEG.:    1
        % PTS CARD.:    4
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.4e-16
        
        xyw=[
            7.88675134594812865529434020572808e-01 1.66666666666666685170383743752609e-01 5.28312163512967558620658792278846e-02
            2.11324865405187134470565979427192e-01 6.22008467928146235870201508078026e-01 1.97168783648703105360056042627548e-01
            7.88675134594812865529434020572808e-01 4.46581987385204562390761395818117e-02 5.28312163512967558620658792278846e-02
            2.11324865405187134470565979427192e-01 1.66666666666666685170383743752609e-01 1.97168783648703105360056042627548e-01
            ];
        
    case   2
        
        % ALG. DEG.:    2
        % PTS CARD.:    4
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.4e-16
        
        xyw=[
            7.88675134594812865529434020572808e-01 1.66666666666666685170383743752609e-01 5.28312163512967558620658792278846e-02
            2.11324865405187134470565979427192e-01 6.22008467928146235870201508078026e-01 1.97168783648703105360056042627548e-01
            7.88675134594812865529434020572808e-01 4.46581987385204562390761395818117e-02 5.28312163512967558620658792278846e-02
            2.11324865405187134470565979427192e-01 1.66666666666666685170383743752609e-01 1.97168783648703105360056042627548e-01
            ];
        
    case   3
        
        % ALG. DEG.:    3
        % PTS CARD.:    9
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.9e-16
        
        xyw=[
            8.87298334620741702138957407441922e-01 1.00000000000000033306690738754696e-01 8.69611615580698042726748298036910e-03
            5.00000000000000000000000000000000e-01 4.43649167310370851069478703720961e-01 6.17283950617283985740080254345230e-02
            1.12701665379258297861042592558078e-01 7.87298334620741724343417899945052e-01 6.84643776713535906486285398386826e-02
            8.87298334620741702138957407441922e-01 5.63508326896291766860969119079527e-02 1.39137858492911561936189457355795e-02
            5.00000000000000000000000000000000e-01 2.50000000000000000000000000000000e-01 9.87654320987653433494557475569309e-02
            1.12701665379258297861042592558078e-01 4.43649167310370851069478703720961e-01 1.09543004274165647893291009040695e-01
            8.87298334620741702138957407441922e-01 1.27016653792583148613326571307880e-02 8.69611615580698042726748298036910e-03
            5.00000000000000000000000000000000e-01 5.63508326896291489305212962790392e-02 6.17283950617283985740080254345230e-02
            1.12701665379258297861042592558078e-01 9.99999999999999916733273153113259e-02 6.84643776713535906486285398386826e-02
            ];
        
    case   4
        
        % ALG. DEG.:    4
        % PTS CARD.:    9
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.9e-16
        
        xyw=[
            8.87298334620741702138957407441922e-01 1.00000000000000033306690738754696e-01 8.69611615580698042726748298036910e-03
            5.00000000000000000000000000000000e-01 4.43649167310370851069478703720961e-01 6.17283950617283985740080254345230e-02
            1.12701665379258297861042592558078e-01 7.87298334620741724343417899945052e-01 6.84643776713535906486285398386826e-02
            8.87298334620741702138957407441922e-01 5.63508326896291766860969119079527e-02 1.39137858492911561936189457355795e-02
            5.00000000000000000000000000000000e-01 2.50000000000000000000000000000000e-01 9.87654320987653433494557475569309e-02
            1.12701665379258297861042592558078e-01 4.43649167310370851069478703720961e-01 1.09543004274165647893291009040695e-01
            8.87298334620741702138957407441922e-01 1.27016653792583148613326571307880e-02 8.69611615580698042726748298036910e-03
            5.00000000000000000000000000000000e-01 5.63508326896291489305212962790392e-02 6.17283950617283985740080254345230e-02
            1.12701665379258297861042592558078e-01 9.99999999999999916733273153113259e-02 6.84643776713535906486285398386826e-02
            ];
        
    case   5
        
        % ALG. DEG.:    5
        % PTS CARD.:   16
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.2e-16
        
        xyw=[
            9.30568155797026230757751363853458e-01 6.46110632135476897763837200727721e-02 2.10036524447485169198035492854615e-03
            6.69990521792428239677974488586187e-01 3.07096311531159016361414160201093e-01 1.87158153150127730401930392645227e-02
            3.30009478207571815833176742671640e-01 6.23471844265867214396337203652365e-01 3.79971476479502301426549593088566e-02
            6.94318442029737692422486361465417e-02 8.65957092583478527103579835966229e-01 2.81503830769257064325916672942185e-02
            9.30568155797026230757751363853458e-01 4.65186775265609420149104380470817e-02 3.93768560873346707557063339777415e-03
            6.69990521792428239677974488586187e-01 2.21103222500738022526789450239448e-01 3.50877052529335042851954540310544e-02
            3.30009478207571815833176742671640e-01 4.48887299291690244906760653975653e-01 7.12356204997400821454789365816396e-02
            6.94318442029737692422486361465417e-02 6.23471844265867214396337203652365e-01 5.27752773542295786080025266073790e-02
            9.30568155797026230757751363853458e-01 2.29131666764127751856339187952472e-02 3.93768560873346534084715742096705e-03
            6.69990521792428239677974488586187e-01 1.08906255706833821061962908061105e-01 3.50877052529334904074076462165976e-02
            3.30009478207571815833176742671640e-01 2.21103222500738022526789450239448e-01 7.12356204997400543899033209527261e-02
            6.94318442029737692422486361465417e-02 3.07096311531159016361414160201093e-01 5.27752773542295508524269109784655e-02
            9.30568155797026230757751363853458e-01 4.82078098942601875054325688552126e-03 2.10036524447485255934209291694970e-03
            6.69990521792428239677974488586187e-01 2.29131666764127925328686785633181e-02 1.87158153150127799790869431717510e-02
            3.30009478207571815833176742671640e-01 4.65186775265609767093799575832236e-02 3.79971476479502509593366710305418e-02
            6.94318442029737692422486361465417e-02 6.46110632135477452875349513305991e-02 2.81503830769257203103794751086753e-02
            ];
        
    case   6
        
        % ALG. DEG.:    6
        % PTS CARD.:   16
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.2e-16
        
        xyw=[
            9.30568155797026230757751363853458e-01 6.46110632135476897763837200727721e-02 2.10036524447485169198035492854615e-03
            6.69990521792428239677974488586187e-01 3.07096311531159016361414160201093e-01 1.87158153150127730401930392645227e-02
            3.30009478207571815833176742671640e-01 6.23471844265867214396337203652365e-01 3.79971476479502301426549593088566e-02
            6.94318442029737692422486361465417e-02 8.65957092583478527103579835966229e-01 2.81503830769257064325916672942185e-02
            9.30568155797026230757751363853458e-01 4.65186775265609420149104380470817e-02 3.93768560873346707557063339777415e-03
            6.69990521792428239677974488586187e-01 2.21103222500738022526789450239448e-01 3.50877052529335042851954540310544e-02
            3.30009478207571815833176742671640e-01 4.48887299291690244906760653975653e-01 7.12356204997400821454789365816396e-02
            6.94318442029737692422486361465417e-02 6.23471844265867214396337203652365e-01 5.27752773542295786080025266073790e-02
            9.30568155797026230757751363853458e-01 2.29131666764127751856339187952472e-02 3.93768560873346534084715742096705e-03
            6.69990521792428239677974488586187e-01 1.08906255706833821061962908061105e-01 3.50877052529334904074076462165976e-02
            3.30009478207571815833176742671640e-01 2.21103222500738022526789450239448e-01 7.12356204997400543899033209527261e-02
            6.94318442029737692422486361465417e-02 3.07096311531159016361414160201093e-01 5.27752773542295508524269109784655e-02
            9.30568155797026230757751363853458e-01 4.82078098942601875054325688552126e-03 2.10036524447485255934209291694970e-03
            6.69990521792428239677974488586187e-01 2.29131666764127925328686785633181e-02 1.87158153150127799790869431717510e-02
            3.30009478207571815833176742671640e-01 4.65186775265609767093799575832236e-02 3.79971476479502509593366710305418e-02
            6.94318442029737692422486361465417e-02 6.46110632135477452875349513305991e-02 2.81503830769257203103794751086753e-02
            ];
        
    case   7
        
        % ALG. DEG.:    7
        % PTS CARD.:   25
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.0e-16
        
        xyw=[
            9.53089922969331926339009442017414e-01 4.47095217036448597247577652069594e-02 6.58316657300728972496139945747018e-04
            7.69234655052841609368385888956254e-01 2.19940124839678508017470903723733e-01 6.54219752925193093889832951504104e-03
            5.00000000000000000000000000000000e-01 4.76544961484665963169504721008707e-01 1.68481340484401068136044443690480e-02
            2.30765344947158446142765342301573e-01 7.33149798129653307299236075778026e-01 2.18078024707480423727279372769772e-02
            4.69100770306680736609905579825863e-02 9.08380401265687087430933388532139e-01 1.33752705583063940392163360115774e-02
            9.53089922969331926339009442017414e-01 3.60848569231881841079534467553458e-02 1.32990068381943788572430165118021e-03
            7.69234655052841609368385888956254e-01 1.77512700518577398733910399641900e-01 1.32162430820271219883421665031165e-02
            5.00000000000000000000000000000000e-01 3.84617327526420804684192944478127e-01 3.40358165688438504825619190796715e-02
            2.30765344947158446142765342301573e-01 5.91721954534264127367748642427614e-01 4.40551079739706502591545245195448e-02
            4.69100770306680736609905579825863e-02 7.33149798129653418321538538293680e-01 2.70200993161805146092202534191529e-02
            9.53089922969331926339009442017414e-01 2.34550385153340368304952789912932e-02 1.58069453207069403680162800185371e-03
            7.69234655052841609368385888956254e-01 1.15382672473579195315807055521873e-01 1.57085739021349223010215467866146e-02
            5.00000000000000000000000000000000e-01 2.50000000000000000000000000000000e-01 4.04543209876543838343643244570558e-02
            2.30765344947158446142765342301573e-01 3.84617327526420749173041713220300e-01 5.23630592355528237669126667697128e-02
            4.69100770306680736609905579825863e-02 4.76544961484665963169504721008707e-01 3.21155735648095130851942258232157e-02
            9.53089922969331926339009442017414e-01 1.08252201074798930224840631808547e-02 1.32990068381943918676690863378553e-03
            7.69234655052841609368385888956254e-01 5.32526444285810127143854231235309e-02 1.32162430820271341314064983407661e-02
            5.00000000000000000000000000000000e-01 1.15382672473579223071382671150786e-01 3.40358165688438851770314386158134e-02
            2.30765344947158446142765342301573e-01 1.77512700518577398733910399641900e-01 4.40551079739706918925179479629151e-02
            4.69100770306680736609905579825863e-02 2.19940124839678563528622134981561e-01 2.70200993161805388953489170944522e-02
            9.53089922969331926339009442017414e-01 2.20055532702321220150931679881978e-03 6.58316657300728864075922697196575e-04
            7.69234655052841609368385888956254e-01 1.08252201074798895530371112272405e-02 6.54219752925192920417485353823395e-03
            5.00000000000000000000000000000000e-01 2.34550385153340368304952789912932e-02 1.68481340484401033441574924154338e-02
            2.30765344947158446142765342301573e-01 3.60848569231881771690595428481174e-02 2.18078024707480389032809853233630e-02
            4.69100770306680736609905579825863e-02 4.47095217036448597247577652069594e-02 1.33752705583063905697693840579632e-02
            ];
        
    case   8
        
        % ALG. DEG.:    8
        % PTS CARD.:   25
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.0e-16
        
        xyw=[
            9.53089922969331926339009442017414e-01 4.47095217036448597247577652069594e-02 6.58316657300728972496139945747018e-04
            7.69234655052841609368385888956254e-01 2.19940124839678508017470903723733e-01 6.54219752925193093889832951504104e-03
            5.00000000000000000000000000000000e-01 4.76544961484665963169504721008707e-01 1.68481340484401068136044443690480e-02
            2.30765344947158446142765342301573e-01 7.33149798129653307299236075778026e-01 2.18078024707480423727279372769772e-02
            4.69100770306680736609905579825863e-02 9.08380401265687087430933388532139e-01 1.33752705583063940392163360115774e-02
            9.53089922969331926339009442017414e-01 3.60848569231881841079534467553458e-02 1.32990068381943788572430165118021e-03
            7.69234655052841609368385888956254e-01 1.77512700518577398733910399641900e-01 1.32162430820271219883421665031165e-02
            5.00000000000000000000000000000000e-01 3.84617327526420804684192944478127e-01 3.40358165688438504825619190796715e-02
            2.30765344947158446142765342301573e-01 5.91721954534264127367748642427614e-01 4.40551079739706502591545245195448e-02
            4.69100770306680736609905579825863e-02 7.33149798129653418321538538293680e-01 2.70200993161805146092202534191529e-02
            9.53089922969331926339009442017414e-01 2.34550385153340368304952789912932e-02 1.58069453207069403680162800185371e-03
            7.69234655052841609368385888956254e-01 1.15382672473579195315807055521873e-01 1.57085739021349223010215467866146e-02
            5.00000000000000000000000000000000e-01 2.50000000000000000000000000000000e-01 4.04543209876543838343643244570558e-02
            2.30765344947158446142765342301573e-01 3.84617327526420749173041713220300e-01 5.23630592355528237669126667697128e-02
            4.69100770306680736609905579825863e-02 4.76544961484665963169504721008707e-01 3.21155735648095130851942258232157e-02
            9.53089922969331926339009442017414e-01 1.08252201074798930224840631808547e-02 1.32990068381943918676690863378553e-03
            7.69234655052841609368385888956254e-01 5.32526444285810127143854231235309e-02 1.32162430820271341314064983407661e-02
            5.00000000000000000000000000000000e-01 1.15382672473579223071382671150786e-01 3.40358165688438851770314386158134e-02
            2.30765344947158446142765342301573e-01 1.77512700518577398733910399641900e-01 4.40551079739706918925179479629151e-02
            4.69100770306680736609905579825863e-02 2.19940124839678563528622134981561e-01 2.70200993161805388953489170944522e-02
            9.53089922969331926339009442017414e-01 2.20055532702321220150931679881978e-03 6.58316657300728864075922697196575e-04
            7.69234655052841609368385888956254e-01 1.08252201074798895530371112272405e-02 6.54219752925192920417485353823395e-03
            5.00000000000000000000000000000000e-01 2.34550385153340368304952789912932e-02 1.68481340484401033441574924154338e-02
            2.30765344947158446142765342301573e-01 3.60848569231881771690595428481174e-02 2.18078024707480389032809853233630e-02
            4.69100770306680736609905579825863e-02 4.47095217036448597247577652069594e-02 1.33752705583063905697693840579632e-02
            ];
        
    case   9
        
        % ALG. DEG.:    9
        % PTS CARD.:   36
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 7.3e-16
        
        xyw=[
            9.66234757101575914006730272376444e-01 3.26251512704345617010837088400876e-02 2.47770041950702644539122898592609e-04
            8.30604693233132240770544285624055e-01 1.63675633088031402539996861378313e-01 2.61746635531116518882943644541683e-03
            6.19309593041598382967549696331844e-01 3.67836302898351286128075798842474e-01 7.62952278389595131558031937402120e-03
            3.80690406958401561521299072410329e-01 5.98398454203224683389805704791797e-01 1.24117565455557094661331873908239e-02
            1.69395306766867703718304483118118e-01 8.02559124013544566977884642255958e-01 1.28343569877846386956266044876429e-02
            3.37652428984240304821184963657288e-02 9.33609605831141387000116083072498e-01 7.09025038029440467235442469018381e-03
            9.66234757101575914006730272376444e-01 2.80455692195877813455151539301369e-02 5.21734568403170711342609067173726e-04
            8.30604693233132240770544285624055e-01 1.40700536812226523197821848043532e-01 5.51165374331171516869565252250140e-03
            6.19309593041598382967549696331844e-01 3.16203238688479437712430808460340e-01 1.60656459733340382933786116836927e-02
            3.80690406958401561521299072410329e-01 5.14401454544652803058113477163715e-01 2.61356957985629176355324432279303e-02
            1.69395306766867703718304483118118e-01 6.89904156420905856350600515725091e-01 2.70255744037300772719678576549995e-02
            3.37652428984240304821184963657288e-02 8.02559124013544455955582179740304e-01 1.49300887746926722043916413440456e-02
            9.66234757101575914006730272376444e-01 2.09111388383737759055769345195586e-02 6.76698664554101951898457123490971e-04
            8.30604693233132240770544285624055e-01 1.04908138496945593787224026982585e-01 7.14870923542391175503629696663666e-03
            6.19309593041598382967549696331844e-01 2.35765221008248176959298803012643e-01 2.08374177862661637272712056301316e-02
            3.80690406958401561521299072410329e-01 3.83544372033350178252675277690287e-01 3.38984447555580048483392374691903e-02
            1.69395306766867703718304483118118e-01 5.14401454544652914080415939679369e-01 3.50526325364729704481270289306849e-02
            3.37652428984240304821184963657288e-02 5.98398454203224683389805704791797e-01 1.93645806648975737918050299413153e-02
            9.66234757101575914006730272376444e-01 1.28541040600503638641205483850172e-02 6.76698664554102493999543366243188e-04
            8.30604693233132240770544285624055e-01 6.44871682699221654422316873933596e-02 7.14870923542391695920672489705794e-03
            6.19309593041598382967549696331844e-01 1.44925185950153412317575885026599e-01 2.08374177862661810745059653982025e-02
            3.80690406958401561521299072410329e-01 2.35765221008248149203723187383730e-01 3.38984447555580326039148530981038e-02
            1.69395306766867703718304483118118e-01 3.16203238688479437712430808460340e-01 3.50526325364729982037026445595984e-02
            3.37652428984240304821184963657288e-02 3.67836302898351230616924567584647e-01 1.93645806648975876695928377557721e-02
            9.66234757101575914006730272376444e-01 5.71967367883635842418232897443886e-03 5.21734568403169843980871078770178e-04
            8.30604693233132240770544285624055e-01 2.86947699546412256232930104715706e-02 5.51165374331170649507827263846593e-03
            6.19309593041598382967549696331844e-01 6.44871682699221515644438795789029e-02 1.60656459733340140072499480083934e-02
            3.80690406958401561521299072410329e-01 1.04908138496945552153860603539215e-01 2.61356957985628794716159717381743e-02
            1.69395306766867703718304483118118e-01 1.40700536812226495442246232414618e-01 2.70255744037300356386044342116293e-02
            3.37652428984240304821184963657288e-02 1.63675633088031347028845630120486e-01 1.49300887746926496529864536455534e-02
            9.66234757101575914006730272376444e-01 1.14009162798957806861377406448810e-03 2.47770041950702915589666019968718e-04
            8.30604693233132240770544285624055e-01 5.71967367883634194430930719477146e-03 2.61746635531116822459551940482925e-03
            6.19309593041598382967549696331844e-01 1.28541040600503239654806009184540e-02 7.62952278389595998919769925805667e-03
            3.80690406958401561521299072410329e-01 2.09111388383737065166378954472748e-02 1.24117565455557250786444711820877e-02
            1.69395306766867703718304483118118e-01 2.80455692195876911398944031361680e-02 1.28343569877846543081378882789068e-02
            3.37652428984240304821184963657288e-02 3.26251512704344576176751502316620e-02 7.09025038029441247861006658581573e-03
            ];
        
    case  10
        
        % ALG. DEG.:   10
        % PTS CARD.:   36
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 7.3e-16
        
        xyw=[
            9.66234757101575914006730272376444e-01 3.26251512704345617010837088400876e-02 2.47770041950702644539122898592609e-04
            8.30604693233132240770544285624055e-01 1.63675633088031402539996861378313e-01 2.61746635531116518882943644541683e-03
            6.19309593041598382967549696331844e-01 3.67836302898351286128075798842474e-01 7.62952278389595131558031937402120e-03
            3.80690406958401561521299072410329e-01 5.98398454203224683389805704791797e-01 1.24117565455557094661331873908239e-02
            1.69395306766867703718304483118118e-01 8.02559124013544566977884642255958e-01 1.28343569877846386956266044876429e-02
            3.37652428984240304821184963657288e-02 9.33609605831141387000116083072498e-01 7.09025038029440467235442469018381e-03
            9.66234757101575914006730272376444e-01 2.80455692195877813455151539301369e-02 5.21734568403170711342609067173726e-04
            8.30604693233132240770544285624055e-01 1.40700536812226523197821848043532e-01 5.51165374331171516869565252250140e-03
            6.19309593041598382967549696331844e-01 3.16203238688479437712430808460340e-01 1.60656459733340382933786116836927e-02
            3.80690406958401561521299072410329e-01 5.14401454544652803058113477163715e-01 2.61356957985629176355324432279303e-02
            1.69395306766867703718304483118118e-01 6.89904156420905856350600515725091e-01 2.70255744037300772719678576549995e-02
            3.37652428984240304821184963657288e-02 8.02559124013544455955582179740304e-01 1.49300887746926722043916413440456e-02
            9.66234757101575914006730272376444e-01 2.09111388383737759055769345195586e-02 6.76698664554101951898457123490971e-04
            8.30604693233132240770544285624055e-01 1.04908138496945593787224026982585e-01 7.14870923542391175503629696663666e-03
            6.19309593041598382967549696331844e-01 2.35765221008248176959298803012643e-01 2.08374177862661637272712056301316e-02
            3.80690406958401561521299072410329e-01 3.83544372033350178252675277690287e-01 3.38984447555580048483392374691903e-02
            1.69395306766867703718304483118118e-01 5.14401454544652914080415939679369e-01 3.50526325364729704481270289306849e-02
            3.37652428984240304821184963657288e-02 5.98398454203224683389805704791797e-01 1.93645806648975737918050299413153e-02
            9.66234757101575914006730272376444e-01 1.28541040600503638641205483850172e-02 6.76698664554102493999543366243188e-04
            8.30604693233132240770544285624055e-01 6.44871682699221654422316873933596e-02 7.14870923542391695920672489705794e-03
            6.19309593041598382967549696331844e-01 1.44925185950153412317575885026599e-01 2.08374177862661810745059653982025e-02
            3.80690406958401561521299072410329e-01 2.35765221008248149203723187383730e-01 3.38984447555580326039148530981038e-02
            1.69395306766867703718304483118118e-01 3.16203238688479437712430808460340e-01 3.50526325364729982037026445595984e-02
            3.37652428984240304821184963657288e-02 3.67836302898351230616924567584647e-01 1.93645806648975876695928377557721e-02
            9.66234757101575914006730272376444e-01 5.71967367883635842418232897443886e-03 5.21734568403169843980871078770178e-04
            8.30604693233132240770544285624055e-01 2.86947699546412256232930104715706e-02 5.51165374331170649507827263846593e-03
            6.19309593041598382967549696331844e-01 6.44871682699221515644438795789029e-02 1.60656459733340140072499480083934e-02
            3.80690406958401561521299072410329e-01 1.04908138496945552153860603539215e-01 2.61356957985628794716159717381743e-02
            1.69395306766867703718304483118118e-01 1.40700536812226495442246232414618e-01 2.70255744037300356386044342116293e-02
            3.37652428984240304821184963657288e-02 1.63675633088031347028845630120486e-01 1.49300887746926496529864536455534e-02
            9.66234757101575914006730272376444e-01 1.14009162798957806861377406448810e-03 2.47770041950702915589666019968718e-04
            8.30604693233132240770544285624055e-01 5.71967367883634194430930719477146e-03 2.61746635531116822459551940482925e-03
            6.19309593041598382967549696331844e-01 1.28541040600503239654806009184540e-02 7.62952278389595998919769925805667e-03
            3.80690406958401561521299072410329e-01 2.09111388383737065166378954472748e-02 1.24117565455557250786444711820877e-02
            1.69395306766867703718304483118118e-01 2.80455692195876911398944031361680e-02 1.28343569877846543081378882789068e-02
            3.37652428984240304821184963657288e-02 3.26251512704344576176751502316620e-02 7.09025038029441247861006658581573e-03
            ];
        
    case  11
        
        % ALG. DEG.:   11
        % PTS CARD.:   49
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.4e-15
        
        xyw=[
            9.74553956171379409667565596464556e-01 2.47985426820925627944269820090994e-02 1.06659360355735631314087752397057e-04
            8.70765592799697119019697311159689e-01 1.25945902810518123304817095231556e-01 1.17014141080755721888162490529339e-03
            7.02922575688698647589092161069857e-01 2.89517979151782289815031390389777e-01 3.67196988822000641369824869286731e-03
            4.99999999999999944488848768742173e-01 4.87276978085689815856085260747932e-01 6.76492884474093249952408868352904e-03
            2.97077424311301407922059070187970e-01 6.85035977019597064341382974816952e-01 8.68834290475824048971542623576170e-03
            1.29234407200302769958000226324657e-01 8.48608053360861314118324116861913e-01 7.88426937775226800708683327911785e-03
            2.54460438286207013547368660510983e-02 9.49755413489286826056456902733771e-01 4.08492975558258017421131569335557e-03
            9.74553956171379409667565596464556e-01 2.21575394388359298014634646278864e-02 2.30398933768029246741543270005081e-04
            8.70765592799697119019697311159689e-01 1.12532875195889128794313194248389e-01 2.52766688745082943046149637211784e-03
            7.02922575688698647589092161069857e-01 2.58684799487837457920846873093979e-01 7.93196156673467434039537238277262e-03
            4.99999999999999944488848768742173e-01 4.35382796399848670532151118095499e-01 1.46131796369907730748227692174623e-02
            2.97077424311301407922059070187970e-01 6.12080793311859494565396744292229e-01 1.87680193729914332467778592672403e-02
            1.29234407200302769958000226324657e-01 7.58232717603808059614323155983584e-01 1.70311096195915057105452206087648e-02
            2.54460438286207013547368660510983e-02 8.48608053360861092073719191830605e-01 8.82401185479181164594830732994524e-03
            9.74553956171379409667565596464556e-01 1.78865986691015381448988108559206e-02 3.14521061065585049321829957946761e-04
            8.70765592799697119019697311159689e-01 9.08417823768389587346661073752330e-02 3.45055620900488937280448098476882e-03
            7.02922575688698647589092161069857e-01 2.08822428275864363511615806601185e-01 1.08280404231931859371229620592203e-02
            4.99999999999999944488848768742173e-01 3.51461287844349379305697311792755e-01 1.99486720263898770211241640026856e-02
            2.97077424311301407922059070187970e-01 4.94100147412834200810749507581932e-01 2.56205064439921845198533389975637e-02
            1.29234407200302769958000226324657e-01 6.12080793311859716610001669323537e-01 2.32494247307211930608783490015412e-02
            2.54460438286207013547368660510983e-02 6.85035977019597064341382974816952e-01 1.20457917319126694977171609934885e-02
            9.74553956171379409667565596464556e-01 1.27230219143103211870693414198286e-02 3.44281351761555531529324092332445e-04
            8.70765592799697119019697311159689e-01 6.46172036001513988567879209767852e-02 3.77705121539607729594378859871995e-03
            7.02922575688698647589092161069857e-01 1.48538712155650648449878303836158e-01 1.18526002080616230799359200887011e-02
            4.99999999999999944488848768742173e-01 2.50000000000000000000000000000000e-01 2.18362349021241040902818753011161e-02
            2.97077424311301407922059070187970e-01 3.51461287844349212772243618019274e-01 2.80447438447181049414602682645636e-02
            1.29234407200302769958000226324657e-01 4.35382796399848559509848655579844e-01 2.54493080585854215824870294682114e-02
            2.54460438286207013547368660510983e-02 4.87276978085689593811480335716624e-01 1.31855763377203092506784187776248e-02
            9.74553956171379409667565596464556e-01 7.55944515951910856604856192575426e-03 3.14521061065584995111721333671539e-04
            8.70765592799697119019697311159689e-01 3.83926248234638667344853502072510e-02 3.45055620900488850544274299636527e-03
            7.02922575688698647589092161069857e-01 8.82549960354370027770798401434149e-02 1.08280404231931824676760101056061e-02
            4.99999999999999944488848768742173e-01 1.48538712155650731716605150722899e-01 1.99486720263898700822302600954572e-02
            2.97077424311301407922059070187970e-01 2.08822428275864363511615806601185e-01 2.56205064439921775809594350903353e-02
            1.29234407200302769958000226324657e-01 2.58684799487837568943149335609633e-01 2.32494247307211861219844450943128e-02
            2.54460438286207013547368660510983e-02 2.89517979151782289815031390389777e-01 1.20457917319126660282702090398743e-02
            9.74553956171379409667565596464556e-01 3.28850438978471170531348022336715e-03 2.30398933768028894375837212216140e-04
            8.70765592799697119019697311159689e-01 1.67015320044136758581565516124101e-02 2.52766688745082552733367542430187e-03
            7.02922575688698647589092161069857e-01 3.83926248234638459178036384855659e-02 7.93196156673466219733104054512296e-03
            4.99999999999999944488848768742173e-01 6.46172036001513988567879209767852e-02 1.46131796369907487886941055421630e-02
            2.97077424311301407922059070187970e-01 9.08417823768389171013026839318627e-02 1.87680193729914054912022436383268e-02
            1.29234407200302769958000226324657e-01 1.12532875195889101038737578619475e-01 1.70311096195914814244165569334655e-02
            2.54460438286207013547368660510983e-02 1.25945902810518040038090248344815e-01 8.82401185479179776816049951548848e-03
            9.74553956171379409667565596464556e-01 6.47501146528084241781042518226741e-04 1.06659360355735671971669220603474e-04
            8.70765592799697119019697311159689e-01 3.28850438978472037893086010740262e-03 1.17014141080755765256249389949517e-03
            7.02922575688698647589092161069857e-01 7.55944515951912331119810772861456e-03 3.67196988822000771474085567547263e-03
            4.99999999999999944488848768742173e-01 1.27230219143103541468153849791634e-02 6.76492884474093510160930264873969e-03
            2.97077424311301407922059070187970e-01 1.78865986691015763088152823456767e-02 8.68834290475824395916237818937589e-03
            1.29234407200302769958000226324657e-01 2.21575394388359818431677439320993e-02 7.88426937775227147653378523273204e-03
            2.54460438286207013547368660510983e-02 2.47985426820926148361312613133123e-02 4.08492975558258190893479167016267e-03
            ];
        
    case  12
        
        % ALG. DEG.:   12
        % PTS CARD.:   49
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.4e-15
        
        xyw=[
            9.74553956171379409667565596464556e-01 2.47985426820925627944269820090994e-02 1.06659360355735631314087752397057e-04
            8.70765592799697119019697311159689e-01 1.25945902810518123304817095231556e-01 1.17014141080755721888162490529339e-03
            7.02922575688698647589092161069857e-01 2.89517979151782289815031390389777e-01 3.67196988822000641369824869286731e-03
            4.99999999999999944488848768742173e-01 4.87276978085689815856085260747932e-01 6.76492884474093249952408868352904e-03
            2.97077424311301407922059070187970e-01 6.85035977019597064341382974816952e-01 8.68834290475824048971542623576170e-03
            1.29234407200302769958000226324657e-01 8.48608053360861314118324116861913e-01 7.88426937775226800708683327911785e-03
            2.54460438286207013547368660510983e-02 9.49755413489286826056456902733771e-01 4.08492975558258017421131569335557e-03
            9.74553956171379409667565596464556e-01 2.21575394388359298014634646278864e-02 2.30398933768029246741543270005081e-04
            8.70765592799697119019697311159689e-01 1.12532875195889128794313194248389e-01 2.52766688745082943046149637211784e-03
            7.02922575688698647589092161069857e-01 2.58684799487837457920846873093979e-01 7.93196156673467434039537238277262e-03
            4.99999999999999944488848768742173e-01 4.35382796399848670532151118095499e-01 1.46131796369907730748227692174623e-02
            2.97077424311301407922059070187970e-01 6.12080793311859494565396744292229e-01 1.87680193729914332467778592672403e-02
            1.29234407200302769958000226324657e-01 7.58232717603808059614323155983584e-01 1.70311096195915057105452206087648e-02
            2.54460438286207013547368660510983e-02 8.48608053360861092073719191830605e-01 8.82401185479181164594830732994524e-03
            9.74553956171379409667565596464556e-01 1.78865986691015381448988108559206e-02 3.14521061065585049321829957946761e-04
            8.70765592799697119019697311159689e-01 9.08417823768389587346661073752330e-02 3.45055620900488937280448098476882e-03
            7.02922575688698647589092161069857e-01 2.08822428275864363511615806601185e-01 1.08280404231931859371229620592203e-02
            4.99999999999999944488848768742173e-01 3.51461287844349379305697311792755e-01 1.99486720263898770211241640026856e-02
            2.97077424311301407922059070187970e-01 4.94100147412834200810749507581932e-01 2.56205064439921845198533389975637e-02
            1.29234407200302769958000226324657e-01 6.12080793311859716610001669323537e-01 2.32494247307211930608783490015412e-02
            2.54460438286207013547368660510983e-02 6.85035977019597064341382974816952e-01 1.20457917319126694977171609934885e-02
            9.74553956171379409667565596464556e-01 1.27230219143103211870693414198286e-02 3.44281351761555531529324092332445e-04
            8.70765592799697119019697311159689e-01 6.46172036001513988567879209767852e-02 3.77705121539607729594378859871995e-03
            7.02922575688698647589092161069857e-01 1.48538712155650648449878303836158e-01 1.18526002080616230799359200887011e-02
            4.99999999999999944488848768742173e-01 2.50000000000000000000000000000000e-01 2.18362349021241040902818753011161e-02
            2.97077424311301407922059070187970e-01 3.51461287844349212772243618019274e-01 2.80447438447181049414602682645636e-02
            1.29234407200302769958000226324657e-01 4.35382796399848559509848655579844e-01 2.54493080585854215824870294682114e-02
            2.54460438286207013547368660510983e-02 4.87276978085689593811480335716624e-01 1.31855763377203092506784187776248e-02
            9.74553956171379409667565596464556e-01 7.55944515951910856604856192575426e-03 3.14521061065584995111721333671539e-04
            8.70765592799697119019697311159689e-01 3.83926248234638667344853502072510e-02 3.45055620900488850544274299636527e-03
            7.02922575688698647589092161069857e-01 8.82549960354370027770798401434149e-02 1.08280404231931824676760101056061e-02
            4.99999999999999944488848768742173e-01 1.48538712155650731716605150722899e-01 1.99486720263898700822302600954572e-02
            2.97077424311301407922059070187970e-01 2.08822428275864363511615806601185e-01 2.56205064439921775809594350903353e-02
            1.29234407200302769958000226324657e-01 2.58684799487837568943149335609633e-01 2.32494247307211861219844450943128e-02
            2.54460438286207013547368660510983e-02 2.89517979151782289815031390389777e-01 1.20457917319126660282702090398743e-02
            9.74553956171379409667565596464556e-01 3.28850438978471170531348022336715e-03 2.30398933768028894375837212216140e-04
            8.70765592799697119019697311159689e-01 1.67015320044136758581565516124101e-02 2.52766688745082552733367542430187e-03
            7.02922575688698647589092161069857e-01 3.83926248234638459178036384855659e-02 7.93196156673466219733104054512296e-03
            4.99999999999999944488848768742173e-01 6.46172036001513988567879209767852e-02 1.46131796369907487886941055421630e-02
            2.97077424311301407922059070187970e-01 9.08417823768389171013026839318627e-02 1.87680193729914054912022436383268e-02
            1.29234407200302769958000226324657e-01 1.12532875195889101038737578619475e-01 1.70311096195914814244165569334655e-02
            2.54460438286207013547368660510983e-02 1.25945902810518040038090248344815e-01 8.82401185479179776816049951548848e-03
            9.74553956171379409667565596464556e-01 6.47501146528084241781042518226741e-04 1.06659360355735671971669220603474e-04
            8.70765592799697119019697311159689e-01 3.28850438978472037893086010740262e-03 1.17014141080755765256249389949517e-03
            7.02922575688698647589092161069857e-01 7.55944515951912331119810772861456e-03 3.67196988822000771474085567547263e-03
            4.99999999999999944488848768742173e-01 1.27230219143103541468153849791634e-02 6.76492884474093510160930264873969e-03
            2.97077424311301407922059070187970e-01 1.78865986691015763088152823456767e-02 8.68834290475824395916237818937589e-03
            1.29234407200302769958000226324657e-01 2.21575394388359818431677439320993e-02 7.88426937775227147653378523273204e-03
            2.54460438286207013547368660510983e-02 2.47985426820926148361312613133123e-02 4.08492975558258190893479167016267e-03
            ];
        
    case  13
        
        % ALG. DEG.:   13
        % PTS CARD.:   64
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.0e-15
        
        xyw=[
            9.80144928248768199097185060963966e-01 1.94608478769852377698867940125638e-02 5.08648050097810434564370163368352e-05
            8.98333238706813363982917053363053e-01 9.96481604529950576454311317320389e-02 5.72162909043210364014309110558543e-04
            7.62766204958164717453428238513879e-01 2.32523501019462675021287623167154e-01 1.88340292937897802252877887951854e-03
            5.91717321247824834529183135600761e-01 4.00176196870765610302811410292634e-01 3.74741731310934038726156991572225e-03
            4.08282678752175054448514401883585e-01 5.79968731378002755327827344444813e-01 5.43106981880251381278323918877504e-03
            2.37233795041835504591176686517429e-01 7.47621427229305357542443744023331e-01 6.05561321731648821015125960798287e-03
            1.01666761293186524994780484121293e-01 8.80496767795773238596268583933124e-01 5.05566374507048638120565442477528e-03
            1.98550717512318564139661702938611e-02 9.60684080371782989082873882580316e-01 2.51093933486321073747515875140834e-03
            9.80144928248768199097185060963966e-01 1.78364709110402225311631241311261e-02 1.11740902020922213419909518883344e-04
            8.98333238706813363982917053363053e-01 9.13306309413408484720875435414200e-02 1.25693983388138461922134414550101e-03
            7.62766204958164717453428238513879e-01 2.13115003430640298498488505174464e-01 4.13750022549354539574961719949897e-03
            5.91717321247824834529183135600761e-01 3.66773901111334954538989450156805e-01 8.23240727522963830486979475153930e-03
            4.08282678752175054448514401883585e-01 5.31559337595478464955078834464075e-01 1.19310914565562767308826153112022e-02
            2.37233795041835504591176686517429e-01 6.85218235276172871195399238786194e-01 1.33031018808121865443228060144065e-02
            1.01666761293186524994780484121293e-01 8.07002607765472612655344164522830e-01 1.11063912872568330836298144959073e-02
            1.98550717512318564139661702938611e-02 8.80496767795773127573966121417470e-01 5.51608575209276003059599702282867e-03
            9.80144928248768199097185060963966e-01 1.51447777288591430705499973896622e-02 1.57629735226826663560886765402813e-04
            8.98333238706813363982917053363053e-01 7.75479696819916103356362668819202e-02 1.77312953115123716693601885197040e-03
            7.62766204958164717453428238513879e-01 1.80953921531883804574292184952355e-01 5.83665473653834782075122689093405e-03
            5.91717321247824834529183135600761e-01 3.11424229421950138352315207157517e-01 1.16132244827491636768312943672754e-02
            4.08282678752175054448514401883585e-01 4.51341975536214634612264262614190e-01 1.68308538167329345414113817014368e-02
            2.37233795041835504591176686517429e-01 5.81812283426280774101257975416956e-01 1.87663101804502324698731285934628e-02
            1.01666761293186524994780484121293e-01 6.85218235276173204262306626333157e-01 1.56674725751545368401806967995071e-02
            1.98550717512318564139661702938611e-02 7.47621427229305579587048669054639e-01 7.78138641146863905412311979148399e-03
            9.80144928248768199097185060963966e-01 1.17485898698222387959155810222001e-02 1.82239520571867596359164065411562e-04
            8.98333238706813363982917053363053e-01 6.01579836523464389630433402089693e-02 2.04995761239995001601332802465549e-03
            7.62766204958164717453428238513879e-01 1.40375345711610310939221335502225e-01 6.74789664145325051441970387600122e-03
            5.91717321247824834529183135600761e-01 2.41587932983123310215489709662506e-01 1.34263275833346653470590226220338e-02
            4.08282678752175054448514401883585e-01 3.50129388264701579824844657196081e-01 1.94585540980759703899583001884821e-02
            2.37233795041835504591176686517429e-01 4.51341975536214412567659337582882e-01 2.16961816580287757127987191552165e-02
            1.01666761293186524994780484121293e-01 5.31559337595478464955078834464075e-01 1.81135411193859632850156060612790e-02
            1.98550717512318564139661702938611e-02 5.79968731378002644305524881929159e-01 8.99624761133996364570464265852934e-03
            9.80144928248768199097185060963966e-01 8.10648188140955863745240606021980e-03 1.82239520571867650569272689686784e-04
            8.98333238706813363982917053363053e-01 4.15087776408401831762517986135208e-02 2.04995761239995044969419701885727e-03
            7.62766204958164717453428238513879e-01 9.68584493302249993629260416128091e-02 6.74789664145325224914317985280832e-03
            5.91717321247824834529183135600761e-01 1.66694745769051827499751539107820e-01 1.34263275833346670817824985988409e-02
            4.08282678752175054448514401883585e-01 2.41587932983123282459914094033593e-01 1.94585540980759738594052521420963e-02
            2.37233795041835504591176686517429e-01 3.11424229421949971818861513384036e-01 2.16961816580287791822456711088307e-02
            1.01666761293186524994780484121293e-01 3.66773901111334899027838218898978e-01 1.81135411193859667544625580148931e-02
            1.98550717512318564139661702938611e-02 4.00176196870765499280508947776980e-01 8.99624761133996538042811863533643e-03
            9.80144928248768199097185060963966e-01 4.71029402237266303643536957679316e-03 1.57629735226826202774963459063429e-04
            8.98333238706813363982917053363053e-01 2.41187916111950534370222953839402e-02 1.77312953115123217960602541865001e-03
            7.62766204958164717453428238513879e-01 5.62798735099515751167942312349624e-02 5.83665473653833134087820511126665e-03
            5.91717321247824834529183135600761e-01 9.68584493302250965074406963140063e-02 1.16132244827491307170852508079406e-02
            4.08282678752175054448514401883585e-01 1.40375345711610449717099413646793e-01 1.68308538167328859691540543508381e-02
            2.37233795041835504591176686517429e-01 1.80953921531883887841019031839096e-01 1.87663101804501769587218973356357e-02
            1.01666761293186524994780484121293e-01 2.13115003430640465031942198947945e-01 1.56674725751544917373703214025227e-02
            1.98550717512318564139661702938611e-02 2.32523501019462841554741316940635e-01 7.78138641146861737007967008139531e-03
            9.80144928248768199097185060963966e-01 2.01860084019157446852399395709199e-03 1.11740902020922213419909518883344e-04
            8.98333238706813363982917053363053e-01 1.03361303518457806061014991882985e-02 1.25693983388138461922134414550101e-03
            7.62766204958164717453428238513879e-01 2.41187916111950118036588719405700e-02 4.13750022549354539574961719949897e-03
            5.91717321247824834529183135600761e-01 4.15087776408401484817822790773789e-02 8.23240727522963830486979475153930e-03
            4.08282678752175054448514401883585e-01 6.01579836523463834518921089511423e-02 1.19310914565562767308826153112022e-02
            2.37233795041835504591176686517429e-01 7.75479696819914993133338043662661e-02 1.33031018808121865443228060144065e-02
            1.01666761293186524994780484121293e-01 9.13306309413407513275728888402227e-02 1.11063912872568330836298144959073e-02
            1.98550717512318564139661702938611e-02 9.96481604529949466231286692163849e-02 5.51608575209276003059599702282867e-03
            9.80144928248768199097185060963966e-01 3.94223874246564162920208884699491e-04 5.08648050097809147074290336831837e-05
            8.98333238706813363982917053363053e-01 2.01860084019158227477963585272391e-03 5.72162909043208954551484879402778e-04
            7.62766204958164717453428238513879e-01 4.71029402237267257741448744923218e-03 1.88340292937897325203921994329903e-03
            5.91717321247824834529183135600761e-01 8.10648188140958465830454571232622e-03 3.74741731310933127996332103748500e-03
            4.08282678752175054448514401883585e-01 1.17485898698222752251085765351490e-02 5.43106981880250080235716936272183e-03
            2.37233795041835504591176686517429e-01 1.51447777288591794997429929026111e-02 6.05561321731647346500171380512256e-03
            1.01666761293186524994780484121293e-01 1.78364709110402780423143553889531e-02 5.05566374507047337077958459872207e-03
            1.98550717512318564139661702938611e-02 1.94608478769852932810380252703908e-02 2.51093933486320466594299283258351e-03
            ];
        
    case  14
        
        % ALG. DEG.:   14
        % PTS CARD.:   64
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.0e-15
        
        xyw=[
            9.80144928248768199097185060963966e-01 1.94608478769852377698867940125638e-02 5.08648050097810434564370163368352e-05
            8.98333238706813363982917053363053e-01 9.96481604529950576454311317320389e-02 5.72162909043210364014309110558543e-04
            7.62766204958164717453428238513879e-01 2.32523501019462675021287623167154e-01 1.88340292937897802252877887951854e-03
            5.91717321247824834529183135600761e-01 4.00176196870765610302811410292634e-01 3.74741731310934038726156991572225e-03
            4.08282678752175054448514401883585e-01 5.79968731378002755327827344444813e-01 5.43106981880251381278323918877504e-03
            2.37233795041835504591176686517429e-01 7.47621427229305357542443744023331e-01 6.05561321731648821015125960798287e-03
            1.01666761293186524994780484121293e-01 8.80496767795773238596268583933124e-01 5.05566374507048638120565442477528e-03
            1.98550717512318564139661702938611e-02 9.60684080371782989082873882580316e-01 2.51093933486321073747515875140834e-03
            9.80144928248768199097185060963966e-01 1.78364709110402225311631241311261e-02 1.11740902020922213419909518883344e-04
            8.98333238706813363982917053363053e-01 9.13306309413408484720875435414200e-02 1.25693983388138461922134414550101e-03
            7.62766204958164717453428238513879e-01 2.13115003430640298498488505174464e-01 4.13750022549354539574961719949897e-03
            5.91717321247824834529183135600761e-01 3.66773901111334954538989450156805e-01 8.23240727522963830486979475153930e-03
            4.08282678752175054448514401883585e-01 5.31559337595478464955078834464075e-01 1.19310914565562767308826153112022e-02
            2.37233795041835504591176686517429e-01 6.85218235276172871195399238786194e-01 1.33031018808121865443228060144065e-02
            1.01666761293186524994780484121293e-01 8.07002607765472612655344164522830e-01 1.11063912872568330836298144959073e-02
            1.98550717512318564139661702938611e-02 8.80496767795773127573966121417470e-01 5.51608575209276003059599702282867e-03
            9.80144928248768199097185060963966e-01 1.51447777288591430705499973896622e-02 1.57629735226826663560886765402813e-04
            8.98333238706813363982917053363053e-01 7.75479696819916103356362668819202e-02 1.77312953115123716693601885197040e-03
            7.62766204958164717453428238513879e-01 1.80953921531883804574292184952355e-01 5.83665473653834782075122689093405e-03
            5.91717321247824834529183135600761e-01 3.11424229421950138352315207157517e-01 1.16132244827491636768312943672754e-02
            4.08282678752175054448514401883585e-01 4.51341975536214634612264262614190e-01 1.68308538167329345414113817014368e-02
            2.37233795041835504591176686517429e-01 5.81812283426280774101257975416956e-01 1.87663101804502324698731285934628e-02
            1.01666761293186524994780484121293e-01 6.85218235276173204262306626333157e-01 1.56674725751545368401806967995071e-02
            1.98550717512318564139661702938611e-02 7.47621427229305579587048669054639e-01 7.78138641146863905412311979148399e-03
            9.80144928248768199097185060963966e-01 1.17485898698222387959155810222001e-02 1.82239520571867596359164065411562e-04
            8.98333238706813363982917053363053e-01 6.01579836523464389630433402089693e-02 2.04995761239995001601332802465549e-03
            7.62766204958164717453428238513879e-01 1.40375345711610310939221335502225e-01 6.74789664145325051441970387600122e-03
            5.91717321247824834529183135600761e-01 2.41587932983123310215489709662506e-01 1.34263275833346653470590226220338e-02
            4.08282678752175054448514401883585e-01 3.50129388264701579824844657196081e-01 1.94585540980759703899583001884821e-02
            2.37233795041835504591176686517429e-01 4.51341975536214412567659337582882e-01 2.16961816580287757127987191552165e-02
            1.01666761293186524994780484121293e-01 5.31559337595478464955078834464075e-01 1.81135411193859632850156060612790e-02
            1.98550717512318564139661702938611e-02 5.79968731378002644305524881929159e-01 8.99624761133996364570464265852934e-03
            9.80144928248768199097185060963966e-01 8.10648188140955863745240606021980e-03 1.82239520571867650569272689686784e-04
            8.98333238706813363982917053363053e-01 4.15087776408401831762517986135208e-02 2.04995761239995044969419701885727e-03
            7.62766204958164717453428238513879e-01 9.68584493302249993629260416128091e-02 6.74789664145325224914317985280832e-03
            5.91717321247824834529183135600761e-01 1.66694745769051827499751539107820e-01 1.34263275833346670817824985988409e-02
            4.08282678752175054448514401883585e-01 2.41587932983123282459914094033593e-01 1.94585540980759738594052521420963e-02
            2.37233795041835504591176686517429e-01 3.11424229421949971818861513384036e-01 2.16961816580287791822456711088307e-02
            1.01666761293186524994780484121293e-01 3.66773901111334899027838218898978e-01 1.81135411193859667544625580148931e-02
            1.98550717512318564139661702938611e-02 4.00176196870765499280508947776980e-01 8.99624761133996538042811863533643e-03
            9.80144928248768199097185060963966e-01 4.71029402237266303643536957679316e-03 1.57629735226826202774963459063429e-04
            8.98333238706813363982917053363053e-01 2.41187916111950534370222953839402e-02 1.77312953115123217960602541865001e-03
            7.62766204958164717453428238513879e-01 5.62798735099515751167942312349624e-02 5.83665473653833134087820511126665e-03
            5.91717321247824834529183135600761e-01 9.68584493302250965074406963140063e-02 1.16132244827491307170852508079406e-02
            4.08282678752175054448514401883585e-01 1.40375345711610449717099413646793e-01 1.68308538167328859691540543508381e-02
            2.37233795041835504591176686517429e-01 1.80953921531883887841019031839096e-01 1.87663101804501769587218973356357e-02
            1.01666761293186524994780484121293e-01 2.13115003430640465031942198947945e-01 1.56674725751544917373703214025227e-02
            1.98550717512318564139661702938611e-02 2.32523501019462841554741316940635e-01 7.78138641146861737007967008139531e-03
            9.80144928248768199097185060963966e-01 2.01860084019157446852399395709199e-03 1.11740902020922213419909518883344e-04
            8.98333238706813363982917053363053e-01 1.03361303518457806061014991882985e-02 1.25693983388138461922134414550101e-03
            7.62766204958164717453428238513879e-01 2.41187916111950118036588719405700e-02 4.13750022549354539574961719949897e-03
            5.91717321247824834529183135600761e-01 4.15087776408401484817822790773789e-02 8.23240727522963830486979475153930e-03
            4.08282678752175054448514401883585e-01 6.01579836523463834518921089511423e-02 1.19310914565562767308826153112022e-02
            2.37233795041835504591176686517429e-01 7.75479696819914993133338043662661e-02 1.33031018808121865443228060144065e-02
            1.01666761293186524994780484121293e-01 9.13306309413407513275728888402227e-02 1.11063912872568330836298144959073e-02
            1.98550717512318564139661702938611e-02 9.96481604529949466231286692163849e-02 5.51608575209276003059599702282867e-03
            9.80144928248768199097185060963966e-01 3.94223874246564162920208884699491e-04 5.08648050097809147074290336831837e-05
            8.98333238706813363982917053363053e-01 2.01860084019158227477963585272391e-03 5.72162909043208954551484879402778e-04
            7.62766204958164717453428238513879e-01 4.71029402237267257741448744923218e-03 1.88340292937897325203921994329903e-03
            5.91717321247824834529183135600761e-01 8.10648188140958465830454571232622e-03 3.74741731310933127996332103748500e-03
            4.08282678752175054448514401883585e-01 1.17485898698222752251085765351490e-02 5.43106981880250080235716936272183e-03
            2.37233795041835504591176686517429e-01 1.51447777288591794997429929026111e-02 6.05561321731647346500171380512256e-03
            1.01666761293186524994780484121293e-01 1.78364709110402780423143553889531e-02 5.05566374507047337077958459872207e-03
            1.98550717512318564139661702938611e-02 1.94608478769852932810380252703908e-02 2.51093933486320466594299283258351e-03
            ];
        
    case  15
        
        % ALG. DEG.:   15
        % PTS CARD.:   81
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.6e-15
        
        xyw=[
            9.84080119753813153948840408702381e-01 1.56664376591339159217053378370110e-02 2.62897965308831336648208443218522e-05
            9.18015553663317884769412557943724e-01 8.06792637689521530131031568089384e-02 3.00925319778305136626805893484971e-04
            8.06685716350295178855844824283849e-01 1.90236743404124192258208836392441e-01 1.02364624989596201404262743039908e-03
            6.62126711701904513418526221357752e-01 3.32494386010004416842633645501337e-01 2.14429742078996914481092694870767e-03
            4.99999999999999944488848768742173e-01 4.92040059876906576974420204351190e-01 3.35500019883100625631144886540369e-03
            3.37873288298095597603776241157902e-01 6.51585733743808570572753069427563e-01 4.20215698994793641907197212503888e-03
            1.93314283649704710121852713200497e-01 7.93843376349689155979660881712334e-01 4.27159748776213647747379553720748e-03
            8.19844463366822817640411358297570e-02 9.03400855984860751135556711233221e-01 3.36959187250115716621978734224285e-03
            1.59198802461869015623108225554461e-02 9.68413682094679262313263734540669e-01 1.62509175435610400684005583116232e-03
            9.84080119753813153948840408702381e-01 1.46146976784569341406561093776872e-02 5.84341941426591462808992627486759e-05
            9.18015553663317884769412557943724e-01 7.52629968955497580251901013070892e-02 6.68865144608883502655438668682564e-04
            8.06685716350295178855844824283849e-01 1.77465519135711513598607780295424e-01 2.27525320059281067550105426278151e-03
            6.62126711701904513418526221357752e-01 3.10172933825021979181713049911195e-01 4.76611873503287371972580999113234e-03
            4.99999999999999944488848768742173e-01 4.59007776831658942384706278971862e-01 7.45714150875421109831986399285597e-03
            3.37873288298095597603776241157902e-01 6.07842619838295794565397045516875e-01 9.34011250638999701578768508625217e-03
            1.93314283649704710121852713200497e-01 7.40550034527606482193107240163954e-01 9.49445754005624059645995771461457e-03
            8.19844463366822817640411358297570e-02 8.42752556767767835310678492533043e-01 7.48957434599988123441471898900090e-03
            1.59198802461869015623108225554461e-02 9.03400855984860973180161636264529e-01 3.61208299813681318543578946389516e-03
            9.84080119753813153948840408702381e-01 1.28423400006061495109976888784331e-02 8.42996461778878364030226366665488e-05
            9.18015553663317884769412557943724e-01 6.61356818226886966627375841198955e-02 9.64933218615013420065007476722485e-04
            8.06685716350295178855844824283849e-01 1.55943871386706345294470565931988e-01 3.28237674172111859594025418118690e-03
            6.62126711701904513418526221357752e-01 2.72557555606378953960700073366752e-01 6.87580497857431024305574496224835e-03
            4.99999999999999944488848768742173e-01 4.03342858175147589427922412141925e-01 1.07579885358168054337868468905981e-02
            3.37873288298095597603776241157902e-01 5.34128160743916113872842288401444e-01 1.34744423381297842579051859956962e-02
            1.93314283649704710121852713200497e-01 6.50741844963589000094827952125343e-01 1.36971070281846057231645019669486e-02
            8.19844463366822817640411358297570e-02 7.40550034527606260148502315132646e-01 1.08047775220340216972836344666575e-02
            1.59198802461869015623108225554461e-02 7.93843376349689044957358419196680e-01 5.21094409148021783795279304740689e-03
            9.84080119753813153948840408702381e-01 1.05409779580958023331715978088141e-02 1.01034794206831832837252538315198e-04
            9.18015553663317884769412557943724e-01 5.42840918636085453807815781601676e-02 1.15649155822526196468746917389581e-03
            8.06685716350295178855844824283849e-01 1.27998550957988344034532701698481e-01 3.93399348212298324151126038827897e-03
            6.62126711701904513418526221357752e-01 2.23714929352727548428347859044152e-01 8.24078833676987476675179777885205e-03
            4.99999999999999944488848768742173e-01 3.31063355850952256709263110678876e-01 1.28936621572774030852759352683279e-02
            3.37873288298095597603776241157902e-01 4.38411782349176881723451515426859e-01 1.61493857970884634178698746609371e-02
            1.93314283649704710121852713200497e-01 5.34128160743916335917447213432752e-01 1.64162538345811043205557666624372e-02
            8.19844463366822817640411358297570e-02 6.07842619838295794565397045516875e-01 1.29497396831975853520591002165929e-02
            1.59198802461869015623108225554461e-02 6.51585733743808681595055531943217e-01 6.24541961653108084612240347155421e-03
            9.84080119753813153948840408702381e-01 7.95994012309342129085631967200243e-03 1.06822402782645359438362664405986e-04
            9.18015553663317884769412557943724e-01 4.09922231683410229208242014919961e-02 1.22273923569900786791075919524019e-03
            8.06685716350295178855844824283849e-01 9.66571418248524244498653956725320e-02 4.15934569462632613284780447315825e-03
            6.62126711701904513418526221357752e-01 1.68936644149047715535161273692211e-01 8.71284806256806231006262208893531e-03
            4.99999999999999944488848768742173e-01 2.49999999999999972244424384371086e-01 1.36322539489559871328516749144910e-02
            3.37873288298095597603776241157902e-01 3.31063355850952145686960648163222e-01 1.70744762519867247785310127028424e-02
            1.93314283649704710121852713200497e-01 4.03342858175147589427922412141925e-01 1.73566313770072665201293915515635e-02
            8.19844463366822817640411358297570e-02 4.59007776831658775851252585198381e-01 1.36915437818093815858233597282378e-02
            1.59198802461869015623108225554461e-02 4.92040059876906521463268973093363e-01 6.60317799487937073105170426856603e-03
            9.84080119753813153948840408702381e-01 5.37890228809104545271146946561203e-03 1.01034794206831765074616757971171e-04
            9.18015553663317884769412557943724e-01 2.77003544730735247469954884991239e-02 1.15649155822526109732573118549226e-03
            8.06685716350295178855844824283849e-01 6.53157326917165603763493209044100e-02 3.93399348212298063942604642306833e-03
            6.62126711701904513418526221357752e-01 1.14158358945367979786489343041467e-01 8.24078833676986782785789387162367e-03
            4.99999999999999944488848768742173e-01 1.68936644149047798801888120578951e-01 1.28936621572773926769350794074853e-02
            3.37873288298095597603776241157902e-01 2.23714929352727576183923474673065e-01 1.61493857970884530095290188000945e-02
            1.93314283649704710121852713200497e-01 2.72557555606379064983002535882406e-01 1.64162538345810939122149108015947e-02
            8.19844463366822817640411358297570e-02 3.10172933825021979181713049911195e-01 1.29497396831975766784417203325575e-02
            1.59198802461869015623108225554461e-02 3.32494386010004527864936108016991e-01 6.24541961653107650931371352953647e-03
            9.84080119753813153948840408702381e-01 3.07754024558069480543842644237884e-03 8.42996461778877279828053881161054e-05
            9.18015553663317884769412557943724e-01 1.58487645139933491789108188640967e-02 9.64933218615012227442617742667608e-04
            8.06685716350295178855844824283849e-01 3.73704122629985244219419371347612e-02 3.28237674172111425913156423916917e-03
            6.62126711701904513418526221357752e-01 6.53157326917164909874102818321262e-02 6.87580497857430156943836507821288e-03
            4.99999999999999944488848768742173e-01 9.66571418248523550609263566002483e-02 1.07579885358167915559990390761413e-02
            3.37873288298095597603776241157902e-01 1.27998550957988205256654623553914e-01 1.34744423381297669106704262276253e-02
            1.93314283649704710121852713200497e-01 1.55943871386706206516592487787420e-01 1.36971070281845883759297421988776e-02
            8.19844463366822817640411358297570e-02 1.77465519135711319309578470893030e-01 1.08047775220340078194958266522008e-02
            1.59198802461869015623108225554461e-02 1.90236743404124025724755142618960e-01 5.21094409148021089905888914017851e-03
            9.84080119753813153948840408702381e-01 1.30518256772991386206739239383978e-03 5.84341941426590243081548581294271e-05
            9.18015553663317884769412557943724e-01 6.72144944113231817411913127102707e-03 6.68865144608882093192614437526800e-04
            8.06685716350295178855844824283849e-01 1.58487645139934081595090020755379e-02 2.27525320059280590501149532656200e-03
            6.62126711701904513418526221357752e-01 2.77003544730735906664875756177935e-02 4.76611873503286331138495413028977e-03
            4.99999999999999944488848768742173e-01 4.09922231683411408820205679148785e-02 7.45714150875419461844684221318857e-03
            3.37873288298095597603776241157902e-01 5.42840918636086841586596563047351e-02 9.34011250638997619910597336456703e-03
            1.93314283649704710121852713200497e-01 6.61356818226888909517668935222900e-02 9.49445754005621977977824599292944e-03
            8.19844463366822817640411358297570e-02 7.52629968955499384364316028950270e-02 7.48957434599986562190343519773705e-03
            1.59198802461869015623108225554461e-02 8.06792637689523611799202740257897e-02 3.61208299813680537918014756826324e-03
            9.84080119753813153948840408702381e-01 2.53442587052931051026100073286784e-04 2.62897965308831472173480003906576e-05
            9.18015553663317884769412557943724e-01 1.30518256772991472942913038224333e-03 3.00925319778305299257131766310636e-04
            8.06685716350295178855844824283849e-01 3.07754024558070824954536526263382e-03 1.02364624989596244772349642460085e-03
            6.62126711701904513418526221357752e-01 5.37890228809106193258449124527942e-03 2.14429742078997044585353393131300e-03
            4.99999999999999944488848768742173e-01 7.95994012309345078115541127772303e-03 3.35500019883100799103492484221078e-03
            3.37873288298095597603776241157902e-01 1.05409779580958370276411173449560e-02 4.20215698994793815379544810184598e-03
            1.93314283649704710121852713200497e-01 1.28423400006061963485315402522247e-02 4.27159748776213821219727151401457e-03
            8.19844463366822817640411358297570e-02 1.46146976784569827129134367282859e-02 3.36959187250115890094326331904995e-03
            1.59198802461869015623108225554461e-02 1.56664376591339714328565690948381e-02 1.62509175435610487420179381956586e-03
            ];
        
    case  16
        
        % ALG. DEG.:   16
        % PTS CARD.:   81
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.6e-15
        
        xyw=[
            9.84080119753813153948840408702381e-01 1.56664376591339159217053378370110e-02 2.62897965308831336648208443218522e-05
            9.18015553663317884769412557943724e-01 8.06792637689521530131031568089384e-02 3.00925319778305136626805893484971e-04
            8.06685716350295178855844824283849e-01 1.90236743404124192258208836392441e-01 1.02364624989596201404262743039908e-03
            6.62126711701904513418526221357752e-01 3.32494386010004416842633645501337e-01 2.14429742078996914481092694870767e-03
            4.99999999999999944488848768742173e-01 4.92040059876906576974420204351190e-01 3.35500019883100625631144886540369e-03
            3.37873288298095597603776241157902e-01 6.51585733743808570572753069427563e-01 4.20215698994793641907197212503888e-03
            1.93314283649704710121852713200497e-01 7.93843376349689155979660881712334e-01 4.27159748776213647747379553720748e-03
            8.19844463366822817640411358297570e-02 9.03400855984860751135556711233221e-01 3.36959187250115716621978734224285e-03
            1.59198802461869015623108225554461e-02 9.68413682094679262313263734540669e-01 1.62509175435610400684005583116232e-03
            9.84080119753813153948840408702381e-01 1.46146976784569341406561093776872e-02 5.84341941426591462808992627486759e-05
            9.18015553663317884769412557943724e-01 7.52629968955497580251901013070892e-02 6.68865144608883502655438668682564e-04
            8.06685716350295178855844824283849e-01 1.77465519135711513598607780295424e-01 2.27525320059281067550105426278151e-03
            6.62126711701904513418526221357752e-01 3.10172933825021979181713049911195e-01 4.76611873503287371972580999113234e-03
            4.99999999999999944488848768742173e-01 4.59007776831658942384706278971862e-01 7.45714150875421109831986399285597e-03
            3.37873288298095597603776241157902e-01 6.07842619838295794565397045516875e-01 9.34011250638999701578768508625217e-03
            1.93314283649704710121852713200497e-01 7.40550034527606482193107240163954e-01 9.49445754005624059645995771461457e-03
            8.19844463366822817640411358297570e-02 8.42752556767767835310678492533043e-01 7.48957434599988123441471898900090e-03
            1.59198802461869015623108225554461e-02 9.03400855984860973180161636264529e-01 3.61208299813681318543578946389516e-03
            9.84080119753813153948840408702381e-01 1.28423400006061495109976888784331e-02 8.42996461778878364030226366665488e-05
            9.18015553663317884769412557943724e-01 6.61356818226886966627375841198955e-02 9.64933218615013420065007476722485e-04
            8.06685716350295178855844824283849e-01 1.55943871386706345294470565931988e-01 3.28237674172111859594025418118690e-03
            6.62126711701904513418526221357752e-01 2.72557555606378953960700073366752e-01 6.87580497857431024305574496224835e-03
            4.99999999999999944488848768742173e-01 4.03342858175147589427922412141925e-01 1.07579885358168054337868468905981e-02
            3.37873288298095597603776241157902e-01 5.34128160743916113872842288401444e-01 1.34744423381297842579051859956962e-02
            1.93314283649704710121852713200497e-01 6.50741844963589000094827952125343e-01 1.36971070281846057231645019669486e-02
            8.19844463366822817640411358297570e-02 7.40550034527606260148502315132646e-01 1.08047775220340216972836344666575e-02
            1.59198802461869015623108225554461e-02 7.93843376349689044957358419196680e-01 5.21094409148021783795279304740689e-03
            9.84080119753813153948840408702381e-01 1.05409779580958023331715978088141e-02 1.01034794206831832837252538315198e-04
            9.18015553663317884769412557943724e-01 5.42840918636085453807815781601676e-02 1.15649155822526196468746917389581e-03
            8.06685716350295178855844824283849e-01 1.27998550957988344034532701698481e-01 3.93399348212298324151126038827897e-03
            6.62126711701904513418526221357752e-01 2.23714929352727548428347859044152e-01 8.24078833676987476675179777885205e-03
            4.99999999999999944488848768742173e-01 3.31063355850952256709263110678876e-01 1.28936621572774030852759352683279e-02
            3.37873288298095597603776241157902e-01 4.38411782349176881723451515426859e-01 1.61493857970884634178698746609371e-02
            1.93314283649704710121852713200497e-01 5.34128160743916335917447213432752e-01 1.64162538345811043205557666624372e-02
            8.19844463366822817640411358297570e-02 6.07842619838295794565397045516875e-01 1.29497396831975853520591002165929e-02
            1.59198802461869015623108225554461e-02 6.51585733743808681595055531943217e-01 6.24541961653108084612240347155421e-03
            9.84080119753813153948840408702381e-01 7.95994012309342129085631967200243e-03 1.06822402782645359438362664405986e-04
            9.18015553663317884769412557943724e-01 4.09922231683410229208242014919961e-02 1.22273923569900786791075919524019e-03
            8.06685716350295178855844824283849e-01 9.66571418248524244498653956725320e-02 4.15934569462632613284780447315825e-03
            6.62126711701904513418526221357752e-01 1.68936644149047715535161273692211e-01 8.71284806256806231006262208893531e-03
            4.99999999999999944488848768742173e-01 2.49999999999999972244424384371086e-01 1.36322539489559871328516749144910e-02
            3.37873288298095597603776241157902e-01 3.31063355850952145686960648163222e-01 1.70744762519867247785310127028424e-02
            1.93314283649704710121852713200497e-01 4.03342858175147589427922412141925e-01 1.73566313770072665201293915515635e-02
            8.19844463366822817640411358297570e-02 4.59007776831658775851252585198381e-01 1.36915437818093815858233597282378e-02
            1.59198802461869015623108225554461e-02 4.92040059876906521463268973093363e-01 6.60317799487937073105170426856603e-03
            9.84080119753813153948840408702381e-01 5.37890228809104545271146946561203e-03 1.01034794206831765074616757971171e-04
            9.18015553663317884769412557943724e-01 2.77003544730735247469954884991239e-02 1.15649155822526109732573118549226e-03
            8.06685716350295178855844824283849e-01 6.53157326917165603763493209044100e-02 3.93399348212298063942604642306833e-03
            6.62126711701904513418526221357752e-01 1.14158358945367979786489343041467e-01 8.24078833676986782785789387162367e-03
            4.99999999999999944488848768742173e-01 1.68936644149047798801888120578951e-01 1.28936621572773926769350794074853e-02
            3.37873288298095597603776241157902e-01 2.23714929352727576183923474673065e-01 1.61493857970884530095290188000945e-02
            1.93314283649704710121852713200497e-01 2.72557555606379064983002535882406e-01 1.64162538345810939122149108015947e-02
            8.19844463366822817640411358297570e-02 3.10172933825021979181713049911195e-01 1.29497396831975766784417203325575e-02
            1.59198802461869015623108225554461e-02 3.32494386010004527864936108016991e-01 6.24541961653107650931371352953647e-03
            9.84080119753813153948840408702381e-01 3.07754024558069480543842644237884e-03 8.42996461778877279828053881161054e-05
            9.18015553663317884769412557943724e-01 1.58487645139933491789108188640967e-02 9.64933218615012227442617742667608e-04
            8.06685716350295178855844824283849e-01 3.73704122629985244219419371347612e-02 3.28237674172111425913156423916917e-03
            6.62126711701904513418526221357752e-01 6.53157326917164909874102818321262e-02 6.87580497857430156943836507821288e-03
            4.99999999999999944488848768742173e-01 9.66571418248523550609263566002483e-02 1.07579885358167915559990390761413e-02
            3.37873288298095597603776241157902e-01 1.27998550957988205256654623553914e-01 1.34744423381297669106704262276253e-02
            1.93314283649704710121852713200497e-01 1.55943871386706206516592487787420e-01 1.36971070281845883759297421988776e-02
            8.19844463366822817640411358297570e-02 1.77465519135711319309578470893030e-01 1.08047775220340078194958266522008e-02
            1.59198802461869015623108225554461e-02 1.90236743404124025724755142618960e-01 5.21094409148021089905888914017851e-03
            9.84080119753813153948840408702381e-01 1.30518256772991386206739239383978e-03 5.84341941426590243081548581294271e-05
            9.18015553663317884769412557943724e-01 6.72144944113231817411913127102707e-03 6.68865144608882093192614437526800e-04
            8.06685716350295178855844824283849e-01 1.58487645139934081595090020755379e-02 2.27525320059280590501149532656200e-03
            6.62126711701904513418526221357752e-01 2.77003544730735906664875756177935e-02 4.76611873503286331138495413028977e-03
            4.99999999999999944488848768742173e-01 4.09922231683411408820205679148785e-02 7.45714150875419461844684221318857e-03
            3.37873288298095597603776241157902e-01 5.42840918636086841586596563047351e-02 9.34011250638997619910597336456703e-03
            1.93314283649704710121852713200497e-01 6.61356818226888909517668935222900e-02 9.49445754005621977977824599292944e-03
            8.19844463366822817640411358297570e-02 7.52629968955499384364316028950270e-02 7.48957434599986562190343519773705e-03
            1.59198802461869015623108225554461e-02 8.06792637689523611799202740257897e-02 3.61208299813680537918014756826324e-03
            9.84080119753813153948840408702381e-01 2.53442587052931051026100073286784e-04 2.62897965308831472173480003906576e-05
            9.18015553663317884769412557943724e-01 1.30518256772991472942913038224333e-03 3.00925319778305299257131766310636e-04
            8.06685716350295178855844824283849e-01 3.07754024558070824954536526263382e-03 1.02364624989596244772349642460085e-03
            6.62126711701904513418526221357752e-01 5.37890228809106193258449124527942e-03 2.14429742078997044585353393131300e-03
            4.99999999999999944488848768742173e-01 7.95994012309345078115541127772303e-03 3.35500019883100799103492484221078e-03
            3.37873288298095597603776241157902e-01 1.05409779580958370276411173449560e-02 4.20215698994793815379544810184598e-03
            1.93314283649704710121852713200497e-01 1.28423400006061963485315402522247e-02 4.27159748776213821219727151401457e-03
            8.19844463366822817640411358297570e-02 1.46146976784569827129134367282859e-02 3.36959187250115890094326331904995e-03
            1.59198802461869015623108225554461e-02 1.56664376591339714328565690948381e-02 1.62509175435610487420179381956586e-03
            ];
        
    case  17
        
        % ALG. DEG.:   17
        % PTS CARD.:  100
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 9.4e-16
        
        xyw=[
            9.86953264258585760693165411794325e-01 1.28765184279079434453985086861394e-02 1.44984073826941992541071993483293e-05
            9.32531683344492212661691610264825e-01 6.65880753571853195582619378001255e-02 1.68065640587604212983705020967307e-04
            8.39704784149512217794608659460209e-01 1.58203886528673454137106091366149e-01 5.85349330761178844942382859528607e-04
            7.16697697064623606699740321346326e-01 2.79606132654044436947060603415594e-01 1.27148723524556019233433534765254e-03
            5.74437169490815580097375914192526e-01 4.20010624718162806079391202729312e-01 2.09621582911699881265898959270544e-03
            4.25562830509184253369170392033993e-01 5.66942639540423010124925440322841e-01 2.82953350526185835531189916025596e-03
            2.83302302935376448811410909911501e-01 7.07347131604541212723802345863078e-01 3.21660630325136314908163193138080e-03
            1.60295215850487782205391340539791e-01 8.28749377729912195533756857912522e-01 3.06634624639910138496623481785264e-03
            6.74683166555078983606108522508293e-02 9.20365188901400288479237588035176e-01 2.32296494856648952365141269638116e-03
            1.30467357414141837956833569478476e-02 9.74076745830677936943686745507875e-01 1.09676863059922249758748602488367e-03
            9.86953264258585760693165411794325e-01 1.21664944430917715267881362706248e-02 3.24998178204728230549121203996066e-05
            9.32531683344492212661691610264825e-01 6.29163429031799198432040043371671e-02 3.76738117284382235602779864791501e-04
            8.39704784149512217794608659460209e-01 1.49480367469124048040995944575116e-01 1.31212664321885496242214852458119e-03
            7.16697697064623606699740321346326e-01 2.64188373451697822158479311838164e-01 2.85018225904329781245083452745348e-03
            5.74437169490815580097375914192526e-01 3.96850822703576577854533979916596e-01 4.69890455968371804906791311395864e-03
            4.25562830509184253369170392033993e-01 5.35680860640915690318308861606056e-01 6.34271896289106355132370040905698e-03
            2.83302302935376448811410909911501e-01 6.68343309892794334992061067168834e-01 7.21038636151411903990604557179722e-03
            1.60295215850487782205391340539791e-01 7.83051315875368136865120050060796e-01 6.87356147140781581494906049556448e-03
            6.74683166555078983606108522508293e-02 8.69615340441312167918397335597547e-01 5.20718832344782209253786220415350e-03
            1.30467357414141837956833569478476e-02 9.20365188901400510523842513066484e-01 2.45853077133361807432576640053412e-03
            9.86953264258585760693165411794325e-01 1.09554064195999702191475222434747e-02 4.76427072033013611655413721557295e-05
            9.32531683344492212661691610264825e-01 5.66534682741441086850642250283272e-02 5.52274597760859099586372433776660e-04
            8.39704784149512217794608659460209e-01 1.34600659625933261054342438001186e-01 1.92349587378755002009533825457765e-03
            7.16697697064623606699740321346326e-01 2.37890299135409960529230488646135e-01 4.17818953920733992396518985401599e-03
            5.74437169490815580097375914192526e-01 3.57347144734770139073987138544908e-01 6.88830120063768733956743872681727e-03
            4.25562830509184253369170392033993e-01 4.82357639414742134231772752173129e-01 9.29803065638999497177774600231714e-03
            2.83302302935376448811410909911501e-01 6.01814485014102173998651323927334e-01 1.05699769808524062647547836490958e-02
            1.60295215850487782205391340539791e-01 7.05104124523578956740266221459024e-01 1.00762126863334997711740825820925e-02
            6.74683166555078983606108522508293e-02 7.83051315875368025842817587545142e-01 7.63341351686583086816328957979749e-03
            1.30467357414141837956833569478476e-02 8.28749377729912306556059320428176e-01 3.60405286995698357146999946110100e-03
            9.86953264258585760693165411794325e-01 9.35056546008230030087027273566491e-03 5.85549703795180880396865907844273e-05
            9.32531683344492212661691610264825e-01 4.83543871718292161965280229196651e-02 6.78769629426232670894869603017696e-04
            8.39704784149512217794608659460209e-01 1.14883212050521307800998727088881e-01 2.36406053573191225064542564382464e-03
            7.16697697064623606699740321346326e-01 2.03042108086888606344189156516222e-01 5.13517763934592374325260877299115e-03
            5.74437169490815580097375914192526e-01 3.04999900582235194956837176505360e-01 8.46602337367993382222230991374090e-03
            4.25562830509184253369170392033993e-01 4.11697796482388467254054376098793e-01 1.14276862427129324850660907486599e-02
            2.83302302935376448811410909911501e-01 5.13655588977734889333248702314449e-01 1.29909638926461509095267032876109e-02
            1.60295215850487782205391340539791e-01 6.01814485014102285020953786442988e-01 1.23841059843278183255632640680233e-02
            6.74683166555078983606108522508293e-02 6.68343309892794334992061067168834e-01 9.38179899113119643294655958243311e-03
            1.30467357414141837956833569478476e-02 7.07347131604541434768407270894386e-01 4.42953856811740577514857619689792e-03
            9.86953264258585760693165411794325e-01 7.49452995039265323917732075642562e-03 6.42649498940880763934307973173077e-05
            9.32531683344492212661691610264825e-01 3.87563088498999452902182838442968e-02 7.44959752211979127183016924362846e-04
            8.39704784149512217794608659460209e-01 9.20795300760734874989665854627674e-02 2.59459155884987939175156945736944e-03
            7.16697697064623606699740321346326e-01 1.62739373008427196110048384980473e-01 5.63593374825174537462446622271273e-03
            5.74437169490815580097375914192526e-01 2.44459107798195601768398432795948e-01 9.29158642529223990913500585975271e-03
            4.25562830509184253369170392033993e-01 3.29978061692620061595704328283318e-01 1.25420554230216366869976596376546e-02
            2.83302302935376448811410909911501e-01 4.11697796482388300720600682325312e-01 1.42577758681411408819839081729697e-02
            1.60295215850487782205391340539791e-01 4.82357639414742078720621520915302e-01 1.35917402981778016185510082891597e-02
            6.74683166555078983606108522508293e-02 5.35680860640915579296006399090402e-01 1.02966637703628283856405190022087e-02
            1.30467357414141837956833569478476e-02 5.66942639540423010124925440322841e-01 4.86148438448477486495402644095520e-03
            9.86953264258585760693165411794325e-01 5.55220579102158433293379147244195e-03 6.42649498940879950782678609044751e-05
            9.32531683344492212661691610264825e-01 2.87120078056078351091962019836501e-02 7.44959752211978259821278935959299e-04
            8.39704784149512217794608659460209e-01 6.82156857744142114396979081902828e-02 2.59459155884987678966635549215880e-03
            7.16697697064623606699740321346326e-01 1.20562929926949155556847870229831e-01 5.63593374825173843573056231548435e-03
            5.74437169490815580097375914192526e-01 1.81103722710988762623074421753699e-01 9.29158642529222776607067402210305e-03
            4.25562830509184253369170392033993e-01 2.44459107798195546257247201538121e-01 1.25420554230216228092098518231978e-02
            2.83302302935376448811410909911501e-01 3.04999900582235028423383482731879e-01 1.42577758681411235347491484048987e-02
            1.60295215850487782205391340539791e-01 3.57347144734770028051684676029254e-01 1.35917402981777860060397244978958e-02
            6.74683166555078983606108522508293e-02 3.96850822703576355809929054885288e-01 1.02966637703628162425761871645591e-02
            1.30467357414141837956833569478476e-02 4.20010624718162695057088740213658e-01 4.86148438448476966078359851053392e-03
            9.86953264258585760693165411794325e-01 3.69617028133193987332605345841330e-03 5.85549703795181354735316370252463e-05
            9.32531683344492212661691610264825e-01 1.91139294836785711417803668155102e-02 6.78769629426233212995955845769913e-04
            8.39704784149512217794608659460209e-01 4.54120037999664327710291900075390e-02 2.36406053573191441904977061483351e-03
            7.16697697064623606699740321346326e-01 8.02601948484877869560705221374519e-02 5.13517763934592808006129871500889e-03
            5.74437169490815580097375914192526e-01 1.20562929926949224945786909302115e-01 8.46602337367994076111621382096928e-03
            4.25562830509184253369170392033993e-01 1.62739373008427251621199616238300e-01 1.14276862427129411586834706326954e-02
            2.83302302935376448811410909911501e-01 2.03042108086888634099764772145136e-01 1.29909638926461613178675591484534e-02
            1.60295215850487782205391340539791e-01 2.37890299135410016040381719903962e-01 1.23841059843278287339041199288658e-02
            6.74683166555078983606108522508293e-02 2.64188373451697822158479311838164e-01 9.38179899113120510656393946646858e-03
            1.30467357414141837956833569478476e-02 2.79606132654044547969363065931248e-01 4.42953856811740924459552815051211e-03
            9.86953264258585760693165411794325e-01 2.09132932181426908768706596219999e-03 4.76427072033013205079599039493132e-05
            9.32531683344492212661691610264825e-01 1.08148483813636803879676406836552e-02 5.52274597760858665905503439574886e-04
            8.39704784149512217794608659460209e-01 2.56945562245544621704507193271638e-02 1.92349587378754871905273127197233e-03
            7.16697697064623606699740321346326e-01 4.54120037999664397099230939147674e-02 4.17818953920733732187997588880535e-03
            5.74437169490815580097375914192526e-01 6.82156857744142669508491394481098e-02 6.88830120063768213539701079639599e-03
            4.25562830509184253369170392033993e-01 9.20795300760735291323300089061377e-02 9.29803065638998803288384209508877e-03
            2.83302302935376448811410909911501e-01 1.14883212050521321678786534903338e-01 1.05699769808523975911374037650603e-02
            1.60295215850487782205391340539791e-01 1.34600659625933316565493669259013e-01 1.00762126863334910975567026980571e-02
            6.74683166555078983606108522508293e-02 1.49480367469124075796571560204029e-01 7.63341351686582479663112366097266e-03
            1.30467357414141837956833569478476e-02 1.58203886528673537403832938252890e-01 3.60405286995698096938478549589036e-03
            9.86953264258585760693165411794325e-01 8.80241298322468538987972674902949e-04 3.24998178204728908175479007436337e-05
            9.32531683344492212661691610264825e-01 4.55197375232787703608350327044718e-03 3.76738117284382994544300604644604e-04
            8.39704784149512217794608659460209e-01 1.08148483813636942657554484981119e-02 1.31212664321885756450736248979183e-03
            7.16697697064623606699740321346326e-01 1.91139294836785988973559824444237e-02 2.85018225904330345030213145207654e-03
            5.74437169490815580097375914192526e-01 2.87120078056078906203474332414771e-02 4.69890455968372759004703098639766e-03
            4.25562830509184253369170392033993e-01 3.87563088499000146791573229165806e-02 6.34271896289107656174977023511019e-03
            2.83302302935376448811410909911501e-01 4.83543871718292925243609658991772e-02 7.21038636151413378505559137465752e-03
            1.60295215850487782205391340539791e-01 5.66534682741441988906849758222961e-02 6.87356147140782969273686831002124e-03
            6.74683166555078983606108522508293e-02 6.29163429031800169877186590383644e-02 5.20718832344783250087871806499606e-03
            1.30467357414141837956833569478476e-02 6.65880753571854305805644003157795e-02 2.45853077133362284481532533675363e-03
            9.86953264258585760693165411794325e-01 1.70217313506295048284450155406944e-04 1.44984073826942060303707773827320e-05
            9.32531683344492212661691610264825e-01 8.80241298322463334817544744481665e-04 1.68065640587604294298867957380139e-04
            8.39704784149512217794608659460209e-01 2.09132932181425954670794808976098e-03 5.85349330761179061782817356629494e-04
            7.16697697064623606699740321346326e-01 3.69617028133192382713390067294768e-03 1.27148723524556062601520434185431e-03
            5.74437169490815580097375914192526e-01 5.55220579102156264889034176235327e-03 2.09621582911699968002072758110899e-03
            4.25562830509184253369170392033993e-01 7.49452995039262201415475317389792e-03 2.82953350526185965635450614286128e-03
            2.83302302935376448811410909911501e-01 9.35056546008225866750684929229465e-03 3.21660630325136445012423891398612e-03
            1.60295215850487782205391340539791e-01 1.09554064195999233816136708696831e-02 3.06634624639910268600884180045796e-03
            6.74683166555078983606108522508293e-02 1.21664944430917194850838569664120e-02 2.32296494856649039101315068478470e-03
            1.30467357414141837956833569478476e-02 1.28765184279078896689707534051195e-02 1.09676863059922293126835501908545e-03
            ];
        
    case  18
        
        % ALG. DEG.:   18
        % PTS CARD.:  100
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 9.4e-16
        
        xyw=[
            9.86953264258585760693165411794325e-01 1.28765184279079434453985086861394e-02 1.44984073826941992541071993483293e-05
            9.32531683344492212661691610264825e-01 6.65880753571853195582619378001255e-02 1.68065640587604212983705020967307e-04
            8.39704784149512217794608659460209e-01 1.58203886528673454137106091366149e-01 5.85349330761178844942382859528607e-04
            7.16697697064623606699740321346326e-01 2.79606132654044436947060603415594e-01 1.27148723524556019233433534765254e-03
            5.74437169490815580097375914192526e-01 4.20010624718162806079391202729312e-01 2.09621582911699881265898959270544e-03
            4.25562830509184253369170392033993e-01 5.66942639540423010124925440322841e-01 2.82953350526185835531189916025596e-03
            2.83302302935376448811410909911501e-01 7.07347131604541212723802345863078e-01 3.21660630325136314908163193138080e-03
            1.60295215850487782205391340539791e-01 8.28749377729912195533756857912522e-01 3.06634624639910138496623481785264e-03
            6.74683166555078983606108522508293e-02 9.20365188901400288479237588035176e-01 2.32296494856648952365141269638116e-03
            1.30467357414141837956833569478476e-02 9.74076745830677936943686745507875e-01 1.09676863059922249758748602488367e-03
            9.86953264258585760693165411794325e-01 1.21664944430917715267881362706248e-02 3.24998178204728230549121203996066e-05
            9.32531683344492212661691610264825e-01 6.29163429031799198432040043371671e-02 3.76738117284382235602779864791501e-04
            8.39704784149512217794608659460209e-01 1.49480367469124048040995944575116e-01 1.31212664321885496242214852458119e-03
            7.16697697064623606699740321346326e-01 2.64188373451697822158479311838164e-01 2.85018225904329781245083452745348e-03
            5.74437169490815580097375914192526e-01 3.96850822703576577854533979916596e-01 4.69890455968371804906791311395864e-03
            4.25562830509184253369170392033993e-01 5.35680860640915690318308861606056e-01 6.34271896289106355132370040905698e-03
            2.83302302935376448811410909911501e-01 6.68343309892794334992061067168834e-01 7.21038636151411903990604557179722e-03
            1.60295215850487782205391340539791e-01 7.83051315875368136865120050060796e-01 6.87356147140781581494906049556448e-03
            6.74683166555078983606108522508293e-02 8.69615340441312167918397335597547e-01 5.20718832344782209253786220415350e-03
            1.30467357414141837956833569478476e-02 9.20365188901400510523842513066484e-01 2.45853077133361807432576640053412e-03
            9.86953264258585760693165411794325e-01 1.09554064195999702191475222434747e-02 4.76427072033013611655413721557295e-05
            9.32531683344492212661691610264825e-01 5.66534682741441086850642250283272e-02 5.52274597760859099586372433776660e-04
            8.39704784149512217794608659460209e-01 1.34600659625933261054342438001186e-01 1.92349587378755002009533825457765e-03
            7.16697697064623606699740321346326e-01 2.37890299135409960529230488646135e-01 4.17818953920733992396518985401599e-03
            5.74437169490815580097375914192526e-01 3.57347144734770139073987138544908e-01 6.88830120063768733956743872681727e-03
            4.25562830509184253369170392033993e-01 4.82357639414742134231772752173129e-01 9.29803065638999497177774600231714e-03
            2.83302302935376448811410909911501e-01 6.01814485014102173998651323927334e-01 1.05699769808524062647547836490958e-02
            1.60295215850487782205391340539791e-01 7.05104124523578956740266221459024e-01 1.00762126863334997711740825820925e-02
            6.74683166555078983606108522508293e-02 7.83051315875368025842817587545142e-01 7.63341351686583086816328957979749e-03
            1.30467357414141837956833569478476e-02 8.28749377729912306556059320428176e-01 3.60405286995698357146999946110100e-03
            9.86953264258585760693165411794325e-01 9.35056546008230030087027273566491e-03 5.85549703795180880396865907844273e-05
            9.32531683344492212661691610264825e-01 4.83543871718292161965280229196651e-02 6.78769629426232670894869603017696e-04
            8.39704784149512217794608659460209e-01 1.14883212050521307800998727088881e-01 2.36406053573191225064542564382464e-03
            7.16697697064623606699740321346326e-01 2.03042108086888606344189156516222e-01 5.13517763934592374325260877299115e-03
            5.74437169490815580097375914192526e-01 3.04999900582235194956837176505360e-01 8.46602337367993382222230991374090e-03
            4.25562830509184253369170392033993e-01 4.11697796482388467254054376098793e-01 1.14276862427129324850660907486599e-02
            2.83302302935376448811410909911501e-01 5.13655588977734889333248702314449e-01 1.29909638926461509095267032876109e-02
            1.60295215850487782205391340539791e-01 6.01814485014102285020953786442988e-01 1.23841059843278183255632640680233e-02
            6.74683166555078983606108522508293e-02 6.68343309892794334992061067168834e-01 9.38179899113119643294655958243311e-03
            1.30467357414141837956833569478476e-02 7.07347131604541434768407270894386e-01 4.42953856811740577514857619689792e-03
            9.86953264258585760693165411794325e-01 7.49452995039265323917732075642562e-03 6.42649498940880763934307973173077e-05
            9.32531683344492212661691610264825e-01 3.87563088498999452902182838442968e-02 7.44959752211979127183016924362846e-04
            8.39704784149512217794608659460209e-01 9.20795300760734874989665854627674e-02 2.59459155884987939175156945736944e-03
            7.16697697064623606699740321346326e-01 1.62739373008427196110048384980473e-01 5.63593374825174537462446622271273e-03
            5.74437169490815580097375914192526e-01 2.44459107798195601768398432795948e-01 9.29158642529223990913500585975271e-03
            4.25562830509184253369170392033993e-01 3.29978061692620061595704328283318e-01 1.25420554230216366869976596376546e-02
            2.83302302935376448811410909911501e-01 4.11697796482388300720600682325312e-01 1.42577758681411408819839081729697e-02
            1.60295215850487782205391340539791e-01 4.82357639414742078720621520915302e-01 1.35917402981778016185510082891597e-02
            6.74683166555078983606108522508293e-02 5.35680860640915579296006399090402e-01 1.02966637703628283856405190022087e-02
            1.30467357414141837956833569478476e-02 5.66942639540423010124925440322841e-01 4.86148438448477486495402644095520e-03
            9.86953264258585760693165411794325e-01 5.55220579102158433293379147244195e-03 6.42649498940879950782678609044751e-05
            9.32531683344492212661691610264825e-01 2.87120078056078351091962019836501e-02 7.44959752211978259821278935959299e-04
            8.39704784149512217794608659460209e-01 6.82156857744142114396979081902828e-02 2.59459155884987678966635549215880e-03
            7.16697697064623606699740321346326e-01 1.20562929926949155556847870229831e-01 5.63593374825173843573056231548435e-03
            5.74437169490815580097375914192526e-01 1.81103722710988762623074421753699e-01 9.29158642529222776607067402210305e-03
            4.25562830509184253369170392033993e-01 2.44459107798195546257247201538121e-01 1.25420554230216228092098518231978e-02
            2.83302302935376448811410909911501e-01 3.04999900582235028423383482731879e-01 1.42577758681411235347491484048987e-02
            1.60295215850487782205391340539791e-01 3.57347144734770028051684676029254e-01 1.35917402981777860060397244978958e-02
            6.74683166555078983606108522508293e-02 3.96850822703576355809929054885288e-01 1.02966637703628162425761871645591e-02
            1.30467357414141837956833569478476e-02 4.20010624718162695057088740213658e-01 4.86148438448476966078359851053392e-03
            9.86953264258585760693165411794325e-01 3.69617028133193987332605345841330e-03 5.85549703795181354735316370252463e-05
            9.32531683344492212661691610264825e-01 1.91139294836785711417803668155102e-02 6.78769629426233212995955845769913e-04
            8.39704784149512217794608659460209e-01 4.54120037999664327710291900075390e-02 2.36406053573191441904977061483351e-03
            7.16697697064623606699740321346326e-01 8.02601948484877869560705221374519e-02 5.13517763934592808006129871500889e-03
            5.74437169490815580097375914192526e-01 1.20562929926949224945786909302115e-01 8.46602337367994076111621382096928e-03
            4.25562830509184253369170392033993e-01 1.62739373008427251621199616238300e-01 1.14276862427129411586834706326954e-02
            2.83302302935376448811410909911501e-01 2.03042108086888634099764772145136e-01 1.29909638926461613178675591484534e-02
            1.60295215850487782205391340539791e-01 2.37890299135410016040381719903962e-01 1.23841059843278287339041199288658e-02
            6.74683166555078983606108522508293e-02 2.64188373451697822158479311838164e-01 9.38179899113120510656393946646858e-03
            1.30467357414141837956833569478476e-02 2.79606132654044547969363065931248e-01 4.42953856811740924459552815051211e-03
            9.86953264258585760693165411794325e-01 2.09132932181426908768706596219999e-03 4.76427072033013205079599039493132e-05
            9.32531683344492212661691610264825e-01 1.08148483813636803879676406836552e-02 5.52274597760858665905503439574886e-04
            8.39704784149512217794608659460209e-01 2.56945562245544621704507193271638e-02 1.92349587378754871905273127197233e-03
            7.16697697064623606699740321346326e-01 4.54120037999664397099230939147674e-02 4.17818953920733732187997588880535e-03
            5.74437169490815580097375914192526e-01 6.82156857744142669508491394481098e-02 6.88830120063768213539701079639599e-03
            4.25562830509184253369170392033993e-01 9.20795300760735291323300089061377e-02 9.29803065638998803288384209508877e-03
            2.83302302935376448811410909911501e-01 1.14883212050521321678786534903338e-01 1.05699769808523975911374037650603e-02
            1.60295215850487782205391340539791e-01 1.34600659625933316565493669259013e-01 1.00762126863334910975567026980571e-02
            6.74683166555078983606108522508293e-02 1.49480367469124075796571560204029e-01 7.63341351686582479663112366097266e-03
            1.30467357414141837956833569478476e-02 1.58203886528673537403832938252890e-01 3.60405286995698096938478549589036e-03
            9.86953264258585760693165411794325e-01 8.80241298322468538987972674902949e-04 3.24998178204728908175479007436337e-05
            9.32531683344492212661691610264825e-01 4.55197375232787703608350327044718e-03 3.76738117284382994544300604644604e-04
            8.39704784149512217794608659460209e-01 1.08148483813636942657554484981119e-02 1.31212664321885756450736248979183e-03
            7.16697697064623606699740321346326e-01 1.91139294836785988973559824444237e-02 2.85018225904330345030213145207654e-03
            5.74437169490815580097375914192526e-01 2.87120078056078906203474332414771e-02 4.69890455968372759004703098639766e-03
            4.25562830509184253369170392033993e-01 3.87563088499000146791573229165806e-02 6.34271896289107656174977023511019e-03
            2.83302302935376448811410909911501e-01 4.83543871718292925243609658991772e-02 7.21038636151413378505559137465752e-03
            1.60295215850487782205391340539791e-01 5.66534682741441988906849758222961e-02 6.87356147140782969273686831002124e-03
            6.74683166555078983606108522508293e-02 6.29163429031800169877186590383644e-02 5.20718832344783250087871806499606e-03
            1.30467357414141837956833569478476e-02 6.65880753571854305805644003157795e-02 2.45853077133362284481532533675363e-03
            9.86953264258585760693165411794325e-01 1.70217313506295048284450155406944e-04 1.44984073826942060303707773827320e-05
            9.32531683344492212661691610264825e-01 8.80241298322463334817544744481665e-04 1.68065640587604294298867957380139e-04
            8.39704784149512217794608659460209e-01 2.09132932181425954670794808976098e-03 5.85349330761179061782817356629494e-04
            7.16697697064623606699740321346326e-01 3.69617028133192382713390067294768e-03 1.27148723524556062601520434185431e-03
            5.74437169490815580097375914192526e-01 5.55220579102156264889034176235327e-03 2.09621582911699968002072758110899e-03
            4.25562830509184253369170392033993e-01 7.49452995039262201415475317389792e-03 2.82953350526185965635450614286128e-03
            2.83302302935376448811410909911501e-01 9.35056546008225866750684929229465e-03 3.21660630325136445012423891398612e-03
            1.60295215850487782205391340539791e-01 1.09554064195999233816136708696831e-02 3.06634624639910268600884180045796e-03
            6.74683166555078983606108522508293e-02 1.21664944430917194850838569664120e-02 2.32296494856649039101315068478470e-03
            1.30467357414141837956833569478476e-02 1.28765184279078896689707534051195e-02 1.09676863059922293126835501908545e-03
            ];
        
    case  19
        
        % ALG. DEG.:   19
        % PTS CARD.:  121
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.1e-15
        
        xyw=[
            9.89114329073028430983072212256957e-01 1.07671730954412552677679215662465e-02 8.43364460776226925050550603524258e-06
            9.43531299884047602866132820054190e-01 5.58540004288162958423491488701984e-02 9.86914656403965149253712518806481e-05
            8.65076002787024678220006990159163e-01 1.33455258979163243759913370922732e-01 3.49807635567635081316056888667276e-04
            7.59548064603405848060901917051524e-01 2.37834454754113411301119640484103e-01 7.80360471041545845742593012772659e-04
            6.34771577976172451940328755881637e-01 3.61252665608499057281477462311159e-01 1.33581860957134035405857730438584e-03
            5.00000000000000111022302462515654e-01 4.94557164536514104469233643612824e-01 1.89916856387290140274759497884816e-03
            3.65228422023827603570822475376190e-01 6.27861663464529318190443518687971e-01 2.32166949655472241650921993993961e-03
            2.40451935396594040916795620432822e-01 7.51279874318915186215406265546335e-01 2.46503021277410085468351219617489e-03
            1.34923997212975321779993009840837e-01 8.55659070093865214978734456963139e-01 2.24281964196156643337465297349809e-03
            5.64687001159524526450184112036368e-02 9.33260328644212044935102312592790e-01 1.64902834086735677540658251416517e-03
            1.08856709269715135057765564852161e-02 9.78347155977587146225005199084990e-01 7.66313696584233999646584845066855e-04
            9.89114329073028430983072212256957e-01 1.02709712398354694601332326442389e-02 1.90251026861937951788292938681835e-05
            9.43531299884047602866132820054190e-01 5.32799860231670321941699342005450e-02 2.22633909227259340528684572468876e-04
            8.65076002787024678220006990159163e-01 1.27305014475910222015997419475752e-01 7.89116271388004813375216528470446e-04
            7.59548064603405848060901917051524e-01 2.26873927164383509813561090595613e-01 1.76038222907180710169461335112828e-03
            6.34771577976172451940328755881637e-01 3.44604447786741541204236227713409e-01 3.01341678469975058837770909292431e-03
            5.00000000000000111022302462515654e-01 4.71765649942023690410763947511441e-01 4.28425415422623023786741214280482e-03
            3.65228422023827603570822475376190e-01 5.98926852097306117173047823598608e-01 5.23735616446342495261534466521880e-03
            2.40451935396594040916795620432822e-01 7.16657372719664231830449807603145e-01 5.56075754952174618889193524751136e-03
            1.34923997212975321779993009840837e-01 8.16226285408137353094559784949524e-01 5.05948210761195029533388023423868e-03
            5.64687001159524526450184112036368e-02 8.90251313860880411077403095987393e-01 3.71997338950823350917085363676051e-03
            1.08856709269715135057765564852161e-02 9.33260328644212155957404775108444e-01 1.72869470382154988979950260841179e-03
            9.89114329073028430983072212256957e-01 9.41693269315949099684814882493811e-03 2.82224873794731869630116050107205e-05
            9.43531299884047602866132820054190e-01 4.88497173788872973698715895807254e-02 3.30262747962394985924011736599937e-04
            8.65076002787024678220006990159163e-01 1.16719512189048343908659433054709e-01 1.17060203971180000828700151771500e-03
            7.59548064603405848060901917051524e-01 2.08009199135289563065143170206284e-01 2.61141114781375787562422985388366e-03
            6.34771577976172451940328755881637e-01 3.15950343428585245231232647711295e-01 4.47020542165051971983391609910541e-03
            5.00000000000000111022302462515654e-01 4.32538001393512228087701032563928e-01 6.35540899791565538218263142766773e-03
            3.65228422023827603570822475376190e-01 5.49125659358439377477623111190042e-01 7.76927308574472310492842197504615e-03
            2.40451935396594040916795620432822e-01 6.57066803651735198421590666839620e-01 8.24901775040485542300761068190695e-03
            1.34923997212975321779993009840837e-01 7.48356490597976375944710980547825e-01 7.50540863216311859851259669085266e-03
            5.64687001159524526450184112036368e-02 8.16226285408137242072257322433870e-01 5.51833563103759031781558874740767e-03
            1.08856709269715135057765564852161e-02 8.55659070093865214978734456963139e-01 2.56440479014972792792814715312488e-03
            9.89114329073028430983072212256957e-01 8.26819028449081797060848941782751e-03 3.53282550134762489022974274455180e-05
            9.43531299884047602866132820054190e-01 4.28906918837417619472240915001748e-02 4.13415246664368114890703020947171e-04
            8.65076002787024678220006990159163e-01 1.02481260951670732906038097098644e-01 1.46533247839495989368885364712014e-03
            7.59548064603405848060901917051524e-01 1.82634802160626275036037213794771e-01 3.26890389690099886915719551438997e-03
            6.34771577976172451940328755881637e-01 2.77408541086354154536763871874427e-01 5.59569944970745442036630379334383e-03
            5.00000000000000111022302462515654e-01 3.79774032301702813008148496010108e-01 7.95555355466676453191077200699510e-03
            3.65228422023827603570822475376190e-01 4.82139523517051693524138045177097e-01 9.72539582184944045861740846703469e-03
            2.40451935396594040916795620432822e-01 5.76913262442779628536015934514580e-01 1.03259290642451779379529241964519e-02
            1.34923997212975321779993009840837e-01 6.57066803651735087399288204323966e-01 9.39509641982357729395047840625921e-03
            5.64687001159524526450184112036368e-02 7.16657372719664009785844882571837e-01 6.90772453192917221947988437591448e-03
            1.08856709269715135057765564852161e-02 7.51279874318915075193103803030681e-01 3.21006242880214707341446356281267e-03
            9.89114329073028430983072212256957e-01 6.90991451164308691235138581987485e-03 3.98142119426008142557028846031386e-05
            9.43531299884047602866132820054190e-01 3.58447258788663764006443557263992e-02 4.65910423391107807351840808252064e-04
            8.65076002787024678220006990159163e-01 8.56459186177330328293422212482255e-02 1.65139936402006822120658302566198e-03
            7.59548064603405848060901917051524e-01 1.52632054459120758416190710704541e-01 3.68398701044146551289437674370220e-03
            6.34771577976172451940328755881637e-01 2.31836621769812456506087983143516e-01 6.30623742307008603585893879994728e-03
            5.00000000000000111022302462515654e-01 3.17385788988086170459013146682992e-01 8.96574413951069064399046482094491e-03
            3.65228422023827603570822475376190e-01 4.02934956206359995434240772738121e-01 1.09603197307394338577690717784208e-02
            2.40451935396594040916795620432822e-01 4.82139523517051804546440507692751e-01 1.16371082611154905150918636991264e-02
            1.34923997212975321779993009840837e-01 5.49125659358439377477623111190042e-01 1.05880791433751366564797180558344e-02
            5.64687001159524526450184112036368e-02 5.98926852097306006150745361082954e-01 7.78486252577208562297705896071420e-03
            1.08856709269715135057765564852161e-02 6.27861663464529318190443518687971e-01 3.61767389418345000975940628507033e-03
            9.89114329073028430983072212256957e-01 5.44283546348578537582563185992512e-03 4.13474480423391870312355145244965e-05
            9.43531299884047602866132820054190e-01 2.82343500579762055058274938801333e-02 4.83852526111048608474518584188218e-04
            8.65076002787024678220006990159163e-01 6.74619986064876747677843127348751e-02 1.71499437184418051377532421497563e-03
            7.59548064603405848060901917051524e-01 1.20225967698297103725124657103152e-01 3.82585649874175653603614044584447e-03
            6.34771577976172451940328755881637e-01 1.82614211011913801785411237688095e-01 6.54908916868573911379547425326564e-03
            5.00000000000000111022302462515654e-01 2.50000000000000000000000000000000e-01 9.31101287409053651744628155029204e-03
            3.65228422023827603570822475376190e-01 3.17385788988086281481315609198646e-01 1.13823991103356326332063375161852e-02
            2.40451935396594040916795620432822e-01 3.79774032301703090563904652299243e-01 1.20852506105918276052557658317710e-02
            1.34923997212975321779993009840837e-01 4.32538001393512450132305957595236e-01 1.09958236239870842804178252549718e-02
            5.64687001159524526450184112036368e-02 4.71765649942023856944217641284922e-01 8.08465578234142737135137224413484e-03
            1.08856709269715135057765564852161e-02 4.94557164536514326513838568644132e-01 3.75698967970344984909636387726550e-03
            9.89114329073028430983072212256957e-01 3.97575641532848297193813991157185e-03 3.98142119426007668218578383623196e-05
            9.43531299884047602866132820054190e-01 2.06239742370860242026697761730247e-02 4.65910423391107319460863189775068e-04
            8.65076002787024678220006990159163e-01 4.92780785952422889506507885926112e-02 1.65139936402006648648310704885489e-03
            7.59548064603405848060901917051524e-01 8.78198809374734212784829878728488e-02 3.68398701044146160976655579588623e-03
            6.34771577976172451940328755881637e-01 1.33391800254015091553583260974847e-01 6.30623742307007996432677288112245e-03
            5.00000000000000111022302462515654e-01 1.82614211011913774029835622059181e-01 8.96574413951068023564960896010234e-03
            3.65228422023827603570822475376190e-01 2.31836621769812512017239214401343e-01 1.09603197307394217147047399407711e-02
            2.40451935396594040916795620432822e-01 2.77408541086354210047915103132254e-01 1.16371082611154783720275318614767e-02
            1.34923997212975321779993009840837e-01 3.15950343428585300742383878969122e-01 1.05880791433751262481388621949918e-02
            5.64687001159524526450184112036368e-02 3.44604447786741541204236227713409e-01 7.78486252577207694935967907667873e-03
            1.08856709269715135057765564852161e-02 3.61252665608499112792628693568986e-01 3.61767389418344610663158533725436e-03
            9.89114329073028430983072212256957e-01 2.61748064248074974527669134261032e-03 3.53282550134763979800961442023777e-05
            9.43531299884047602866132820054190e-01 1.35780082322106282477491845384066e-02 4.13415246664369849614178997754266e-04
            8.65076002787024678220006990159163e-01 3.24427362613045749961671049277356e-02 1.46533247839496618206145406304586e-03
            7.59548064603405848060901917051524e-01 5.78171332359678630252730613392487e-02 3.26890389690101231326413433464495e-03
            6.34771577976172451940328755881637e-01 8.78198809374733657673317566150217e-02 5.59569944970747783913322948023961e-03
            5.00000000000000111022302462515654e-01 1.20225967698296992702822194587498e-01 7.95555355466679749165681556632990e-03
            3.65228422023827603570822475376190e-01 1.52632054459120675149463863817800e-01 9.72539582184948209198083191040496e-03
            2.40451935396594040916795620432822e-01 1.82634802160626219524885982536944e-01 1.03259290642452213060398236166293e-02
            1.34923997212975321779993009840837e-01 2.08009199135289452042840707690630e-01 9.39509641982361545786694989601529e-03
            5.64687001159524526450184112036368e-02 2.26873927164383398791258628079959e-01 6.90772453192920084241723799323154e-03
            1.08856709269715135057765564852161e-02 2.37834454754113300278817177968449e-01 3.21006242880216051752140238306765e-03
            9.89114329073028430983072212256957e-01 1.46873823381207845376050791230682e-03 2.82224873794730649902672003914716e-05
            9.43531299884047602866132820054190e-01 7.61898273706510063135732835348790e-03 3.30262747962393576461187505444173e-04
            8.65076002787024678220006990159163e-01 1.82044850239269709324396728788997e-02 1.17060203971179502095700808439460e-03
            7.59548064603405848060901917051524e-01 3.24427362613045888739549127421924e-02 2.61141114781374659992163600463755e-03
            6.34771577976172451940328755881637e-01 4.92780785952422820117568846853828e-02 4.47020542165050063787568035422737e-03
            5.00000000000000111022302462515654e-01 6.74619986064876470122086971059616e-02 6.35540899791562762660701579875422e-03
            3.65228422023827603570822475376190e-01 8.56459186177330328293422212482255e-02 7.76927308574468927782064042730781e-03
            2.40451935396594040916795620432822e-01 1.02481260951670746783825904913101e-01 8.24901775040481899381461516895797e-03
            1.34923997212975321779993009840837e-01 1.16719512189048343908659433054709e-01 7.50540863216308650612829111992141e-03
            5.64687001159524526450184112036368e-02 1.27305014475910194260421803846839e-01 5.51833563103756689904866306051190e-03
            1.08856709269715135057765564852161e-02 1.33455258979163243759913370922732e-01 2.56440479014971708590642229808054e-03
            9.89114329073028430983072212256957e-01 6.14699687136099665214772347354710e-04 1.90251026861938290601471840401970e-05
            9.43531299884047602866132820054190e-01 3.18871409278536537337811473946658e-03 2.22633909227259747104499254533039e-04
            8.65076002787024678220006990159163e-01 7.61898273706510757025123226071628e-03 7.89116271388006222838040759626210e-04
            7.59548064603405848060901917051524e-01 1.35780082322106473297074202832846e-02 1.76038222907181013746069631054070e-03
            6.34771577976172451940328755881637e-01 2.06239742370860415499045359410957e-02 3.01341678469975579254813702334559e-03
            5.00000000000000111022302462515654e-01 2.82343500579762193836153016945900e-02 4.28425415422623804412305403843675e-03
            3.65228422023827603570822475376190e-01 3.58447258788664110951138752625411e-02 5.23735616446343362623272454925427e-03
            2.40451935396594040916795620432822e-01 4.28906918837418105194814188507735e-02 5.56075754952175572987105311995037e-03
            1.34923997212975321779993009840837e-01 4.88497173788873459421289169313241e-02 5.05948210761195896895126011827415e-03
            5.64687001159524526450184112036368e-02 5.32799860231670807664272615511436e-02 3.71997338950824001438388854978712e-03
            1.08856709269715135057765564852161e-02 5.58540004288163513535003801280254e-02 1.72869470382155292556558556782420e-03
            9.89114329073028430983072212256957e-01 1.18497831530313451004268743282921e-04 8.43364460776223706325351037182969e-06
            9.43531299884047602866132820054190e-01 6.14699687136095978927385896639635e-04 9.86914656403961219020837258852907e-05
            8.65076002787024678220006990159163e-01 1.46873823381207108118573501087667e-03 3.49807635567633726063341281786734e-04
            7.59548064603405848060901917051524e-01 2.61748064248073760221235950496066e-03 7.80360471041542809976510053360244e-04
            6.34771577976172451940328755881637e-01 3.97575641532846215525642818988672e-03 1.33581860957133493304771487686367e-03
            5.00000000000000111022302462515654e-01 5.44283546348575588552654025420452e-03 1.89916856387289381333238758031712e-03
            3.65228422023827603570822475376190e-01 6.90991451164305135052012829532941e-03 2.32166949655471287553010206750059e-03
            2.40451935396594040916795620432822e-01 8.26819028449077633724506597445725e-03 2.46503021277409088002352532953410e-03
            1.34923997212975321779993009840837e-01 9.41693269315944242459082147433946e-03 2.24281964196155775975727308946261e-03
            5.64687001159524526450184112036368e-02 1.02709712398354156837054773632190e-02 1.64902834086735005335311310403767e-03
            1.08856709269715135057765564852161e-02 1.07671730954411997566166903084195e-02 7.66313696584230963880501885654439e-04
            ];
        
    case  20
        
        % ALG. DEG.:   20
        % PTS CARD.:  121
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.1e-15
        
        xyw=[
            9.89114329073028430983072212256957e-01 1.07671730954412552677679215662465e-02 8.43364460776226925050550603524258e-06
            9.43531299884047602866132820054190e-01 5.58540004288162958423491488701984e-02 9.86914656403965149253712518806481e-05
            8.65076002787024678220006990159163e-01 1.33455258979163243759913370922732e-01 3.49807635567635081316056888667276e-04
            7.59548064603405848060901917051524e-01 2.37834454754113411301119640484103e-01 7.80360471041545845742593012772659e-04
            6.34771577976172451940328755881637e-01 3.61252665608499057281477462311159e-01 1.33581860957134035405857730438584e-03
            5.00000000000000111022302462515654e-01 4.94557164536514104469233643612824e-01 1.89916856387290140274759497884816e-03
            3.65228422023827603570822475376190e-01 6.27861663464529318190443518687971e-01 2.32166949655472241650921993993961e-03
            2.40451935396594040916795620432822e-01 7.51279874318915186215406265546335e-01 2.46503021277410085468351219617489e-03
            1.34923997212975321779993009840837e-01 8.55659070093865214978734456963139e-01 2.24281964196156643337465297349809e-03
            5.64687001159524526450184112036368e-02 9.33260328644212044935102312592790e-01 1.64902834086735677540658251416517e-03
            1.08856709269715135057765564852161e-02 9.78347155977587146225005199084990e-01 7.66313696584233999646584845066855e-04
            9.89114329073028430983072212256957e-01 1.02709712398354694601332326442389e-02 1.90251026861937951788292938681835e-05
            9.43531299884047602866132820054190e-01 5.32799860231670321941699342005450e-02 2.22633909227259340528684572468876e-04
            8.65076002787024678220006990159163e-01 1.27305014475910222015997419475752e-01 7.89116271388004813375216528470446e-04
            7.59548064603405848060901917051524e-01 2.26873927164383509813561090595613e-01 1.76038222907180710169461335112828e-03
            6.34771577976172451940328755881637e-01 3.44604447786741541204236227713409e-01 3.01341678469975058837770909292431e-03
            5.00000000000000111022302462515654e-01 4.71765649942023690410763947511441e-01 4.28425415422623023786741214280482e-03
            3.65228422023827603570822475376190e-01 5.98926852097306117173047823598608e-01 5.23735616446342495261534466521880e-03
            2.40451935396594040916795620432822e-01 7.16657372719664231830449807603145e-01 5.56075754952174618889193524751136e-03
            1.34923997212975321779993009840837e-01 8.16226285408137353094559784949524e-01 5.05948210761195029533388023423868e-03
            5.64687001159524526450184112036368e-02 8.90251313860880411077403095987393e-01 3.71997338950823350917085363676051e-03
            1.08856709269715135057765564852161e-02 9.33260328644212155957404775108444e-01 1.72869470382154988979950260841179e-03
            9.89114329073028430983072212256957e-01 9.41693269315949099684814882493811e-03 2.82224873794731869630116050107205e-05
            9.43531299884047602866132820054190e-01 4.88497173788872973698715895807254e-02 3.30262747962394985924011736599937e-04
            8.65076002787024678220006990159163e-01 1.16719512189048343908659433054709e-01 1.17060203971180000828700151771500e-03
            7.59548064603405848060901917051524e-01 2.08009199135289563065143170206284e-01 2.61141114781375787562422985388366e-03
            6.34771577976172451940328755881637e-01 3.15950343428585245231232647711295e-01 4.47020542165051971983391609910541e-03
            5.00000000000000111022302462515654e-01 4.32538001393512228087701032563928e-01 6.35540899791565538218263142766773e-03
            3.65228422023827603570822475376190e-01 5.49125659358439377477623111190042e-01 7.76927308574472310492842197504615e-03
            2.40451935396594040916795620432822e-01 6.57066803651735198421590666839620e-01 8.24901775040485542300761068190695e-03
            1.34923997212975321779993009840837e-01 7.48356490597976375944710980547825e-01 7.50540863216311859851259669085266e-03
            5.64687001159524526450184112036368e-02 8.16226285408137242072257322433870e-01 5.51833563103759031781558874740767e-03
            1.08856709269715135057765564852161e-02 8.55659070093865214978734456963139e-01 2.56440479014972792792814715312488e-03
            9.89114329073028430983072212256957e-01 8.26819028449081797060848941782751e-03 3.53282550134762489022974274455180e-05
            9.43531299884047602866132820054190e-01 4.28906918837417619472240915001748e-02 4.13415246664368114890703020947171e-04
            8.65076002787024678220006990159163e-01 1.02481260951670732906038097098644e-01 1.46533247839495989368885364712014e-03
            7.59548064603405848060901917051524e-01 1.82634802160626275036037213794771e-01 3.26890389690099886915719551438997e-03
            6.34771577976172451940328755881637e-01 2.77408541086354154536763871874427e-01 5.59569944970745442036630379334383e-03
            5.00000000000000111022302462515654e-01 3.79774032301702813008148496010108e-01 7.95555355466676453191077200699510e-03
            3.65228422023827603570822475376190e-01 4.82139523517051693524138045177097e-01 9.72539582184944045861740846703469e-03
            2.40451935396594040916795620432822e-01 5.76913262442779628536015934514580e-01 1.03259290642451779379529241964519e-02
            1.34923997212975321779993009840837e-01 6.57066803651735087399288204323966e-01 9.39509641982357729395047840625921e-03
            5.64687001159524526450184112036368e-02 7.16657372719664009785844882571837e-01 6.90772453192917221947988437591448e-03
            1.08856709269715135057765564852161e-02 7.51279874318915075193103803030681e-01 3.21006242880214707341446356281267e-03
            9.89114329073028430983072212256957e-01 6.90991451164308691235138581987485e-03 3.98142119426008142557028846031386e-05
            9.43531299884047602866132820054190e-01 3.58447258788663764006443557263992e-02 4.65910423391107807351840808252064e-04
            8.65076002787024678220006990159163e-01 8.56459186177330328293422212482255e-02 1.65139936402006822120658302566198e-03
            7.59548064603405848060901917051524e-01 1.52632054459120758416190710704541e-01 3.68398701044146551289437674370220e-03
            6.34771577976172451940328755881637e-01 2.31836621769812456506087983143516e-01 6.30623742307008603585893879994728e-03
            5.00000000000000111022302462515654e-01 3.17385788988086170459013146682992e-01 8.96574413951069064399046482094491e-03
            3.65228422023827603570822475376190e-01 4.02934956206359995434240772738121e-01 1.09603197307394338577690717784208e-02
            2.40451935396594040916795620432822e-01 4.82139523517051804546440507692751e-01 1.16371082611154905150918636991264e-02
            1.34923997212975321779993009840837e-01 5.49125659358439377477623111190042e-01 1.05880791433751366564797180558344e-02
            5.64687001159524526450184112036368e-02 5.98926852097306006150745361082954e-01 7.78486252577208562297705896071420e-03
            1.08856709269715135057765564852161e-02 6.27861663464529318190443518687971e-01 3.61767389418345000975940628507033e-03
            9.89114329073028430983072212256957e-01 5.44283546348578537582563185992512e-03 4.13474480423391870312355145244965e-05
            9.43531299884047602866132820054190e-01 2.82343500579762055058274938801333e-02 4.83852526111048608474518584188218e-04
            8.65076002787024678220006990159163e-01 6.74619986064876747677843127348751e-02 1.71499437184418051377532421497563e-03
            7.59548064603405848060901917051524e-01 1.20225967698297103725124657103152e-01 3.82585649874175653603614044584447e-03
            6.34771577976172451940328755881637e-01 1.82614211011913801785411237688095e-01 6.54908916868573911379547425326564e-03
            5.00000000000000111022302462515654e-01 2.50000000000000000000000000000000e-01 9.31101287409053651744628155029204e-03
            3.65228422023827603570822475376190e-01 3.17385788988086281481315609198646e-01 1.13823991103356326332063375161852e-02
            2.40451935396594040916795620432822e-01 3.79774032301703090563904652299243e-01 1.20852506105918276052557658317710e-02
            1.34923997212975321779993009840837e-01 4.32538001393512450132305957595236e-01 1.09958236239870842804178252549718e-02
            5.64687001159524526450184112036368e-02 4.71765649942023856944217641284922e-01 8.08465578234142737135137224413484e-03
            1.08856709269715135057765564852161e-02 4.94557164536514326513838568644132e-01 3.75698967970344984909636387726550e-03
            9.89114329073028430983072212256957e-01 3.97575641532848297193813991157185e-03 3.98142119426007668218578383623196e-05
            9.43531299884047602866132820054190e-01 2.06239742370860242026697761730247e-02 4.65910423391107319460863189775068e-04
            8.65076002787024678220006990159163e-01 4.92780785952422889506507885926112e-02 1.65139936402006648648310704885489e-03
            7.59548064603405848060901917051524e-01 8.78198809374734212784829878728488e-02 3.68398701044146160976655579588623e-03
            6.34771577976172451940328755881637e-01 1.33391800254015091553583260974847e-01 6.30623742307007996432677288112245e-03
            5.00000000000000111022302462515654e-01 1.82614211011913774029835622059181e-01 8.96574413951068023564960896010234e-03
            3.65228422023827603570822475376190e-01 2.31836621769812512017239214401343e-01 1.09603197307394217147047399407711e-02
            2.40451935396594040916795620432822e-01 2.77408541086354210047915103132254e-01 1.16371082611154783720275318614767e-02
            1.34923997212975321779993009840837e-01 3.15950343428585300742383878969122e-01 1.05880791433751262481388621949918e-02
            5.64687001159524526450184112036368e-02 3.44604447786741541204236227713409e-01 7.78486252577207694935967907667873e-03
            1.08856709269715135057765564852161e-02 3.61252665608499112792628693568986e-01 3.61767389418344610663158533725436e-03
            9.89114329073028430983072212256957e-01 2.61748064248074974527669134261032e-03 3.53282550134763979800961442023777e-05
            9.43531299884047602866132820054190e-01 1.35780082322106282477491845384066e-02 4.13415246664369849614178997754266e-04
            8.65076002787024678220006990159163e-01 3.24427362613045749961671049277356e-02 1.46533247839496618206145406304586e-03
            7.59548064603405848060901917051524e-01 5.78171332359678630252730613392487e-02 3.26890389690101231326413433464495e-03
            6.34771577976172451940328755881637e-01 8.78198809374733657673317566150217e-02 5.59569944970747783913322948023961e-03
            5.00000000000000111022302462515654e-01 1.20225967698296992702822194587498e-01 7.95555355466679749165681556632990e-03
            3.65228422023827603570822475376190e-01 1.52632054459120675149463863817800e-01 9.72539582184948209198083191040496e-03
            2.40451935396594040916795620432822e-01 1.82634802160626219524885982536944e-01 1.03259290642452213060398236166293e-02
            1.34923997212975321779993009840837e-01 2.08009199135289452042840707690630e-01 9.39509641982361545786694989601529e-03
            5.64687001159524526450184112036368e-02 2.26873927164383398791258628079959e-01 6.90772453192920084241723799323154e-03
            1.08856709269715135057765564852161e-02 2.37834454754113300278817177968449e-01 3.21006242880216051752140238306765e-03
            9.89114329073028430983072212256957e-01 1.46873823381207845376050791230682e-03 2.82224873794730649902672003914716e-05
            9.43531299884047602866132820054190e-01 7.61898273706510063135732835348790e-03 3.30262747962393576461187505444173e-04
            8.65076002787024678220006990159163e-01 1.82044850239269709324396728788997e-02 1.17060203971179502095700808439460e-03
            7.59548064603405848060901917051524e-01 3.24427362613045888739549127421924e-02 2.61141114781374659992163600463755e-03
            6.34771577976172451940328755881637e-01 4.92780785952422820117568846853828e-02 4.47020542165050063787568035422737e-03
            5.00000000000000111022302462515654e-01 6.74619986064876470122086971059616e-02 6.35540899791562762660701579875422e-03
            3.65228422023827603570822475376190e-01 8.56459186177330328293422212482255e-02 7.76927308574468927782064042730781e-03
            2.40451935396594040916795620432822e-01 1.02481260951670746783825904913101e-01 8.24901775040481899381461516895797e-03
            1.34923997212975321779993009840837e-01 1.16719512189048343908659433054709e-01 7.50540863216308650612829111992141e-03
            5.64687001159524526450184112036368e-02 1.27305014475910194260421803846839e-01 5.51833563103756689904866306051190e-03
            1.08856709269715135057765564852161e-02 1.33455258979163243759913370922732e-01 2.56440479014971708590642229808054e-03
            9.89114329073028430983072212256957e-01 6.14699687136099665214772347354710e-04 1.90251026861938290601471840401970e-05
            9.43531299884047602866132820054190e-01 3.18871409278536537337811473946658e-03 2.22633909227259747104499254533039e-04
            8.65076002787024678220006990159163e-01 7.61898273706510757025123226071628e-03 7.89116271388006222838040759626210e-04
            7.59548064603405848060901917051524e-01 1.35780082322106473297074202832846e-02 1.76038222907181013746069631054070e-03
            6.34771577976172451940328755881637e-01 2.06239742370860415499045359410957e-02 3.01341678469975579254813702334559e-03
            5.00000000000000111022302462515654e-01 2.82343500579762193836153016945900e-02 4.28425415422623804412305403843675e-03
            3.65228422023827603570822475376190e-01 3.58447258788664110951138752625411e-02 5.23735616446343362623272454925427e-03
            2.40451935396594040916795620432822e-01 4.28906918837418105194814188507735e-02 5.56075754952175572987105311995037e-03
            1.34923997212975321779993009840837e-01 4.88497173788873459421289169313241e-02 5.05948210761195896895126011827415e-03
            5.64687001159524526450184112036368e-02 5.32799860231670807664272615511436e-02 3.71997338950824001438388854978712e-03
            1.08856709269715135057765564852161e-02 5.58540004288163513535003801280254e-02 1.72869470382155292556558556782420e-03
            9.89114329073028430983072212256957e-01 1.18497831530313451004268743282921e-04 8.43364460776223706325351037182969e-06
            9.43531299884047602866132820054190e-01 6.14699687136095978927385896639635e-04 9.86914656403961219020837258852907e-05
            8.65076002787024678220006990159163e-01 1.46873823381207108118573501087667e-03 3.49807635567633726063341281786734e-04
            7.59548064603405848060901917051524e-01 2.61748064248073760221235950496066e-03 7.80360471041542809976510053360244e-04
            6.34771577976172451940328755881637e-01 3.97575641532846215525642818988672e-03 1.33581860957133493304771487686367e-03
            5.00000000000000111022302462515654e-01 5.44283546348575588552654025420452e-03 1.89916856387289381333238758031712e-03
            3.65228422023827603570822475376190e-01 6.90991451164305135052012829532941e-03 2.32166949655471287553010206750059e-03
            2.40451935396594040916795620432822e-01 8.26819028449077633724506597445725e-03 2.46503021277409088002352532953410e-03
            1.34923997212975321779993009840837e-01 9.41693269315944242459082147433946e-03 2.24281964196155775975727308946261e-03
            5.64687001159524526450184112036368e-02 1.02709712398354156837054773632190e-02 1.64902834086735005335311310403767e-03
            1.08856709269715135057765564852161e-02 1.07671730954411997566166903084195e-02 7.66313696584230963880501885654439e-04
            ];
        
    case  21
        
        % ALG. DEG.:   21
        % PTS CARD.:  144
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.3e-15
        
        xyw=[
            9.90780317123359566267026821151376e-01 9.13468032429461639087886482002432e-03 5.12962955664186890780004146606075e-06
            9.52058628185237454388811784156132e-01 4.74993675699593789762964490819286e-02 6.04648408077806208253159037102620e-05
            8.84951337097152235244834628247190e-01 1.13987950715501940024410032492597e-01 2.17204656177105492670040276514953e-04
            7.93658977143308685953115855227225e-01 2.04438624061510948459741143778956e-01 4.94418455154532168922820734024981e-04
            6.83915749499090064311701553378953e-01 3.13170053948990945169583710594452e-01 8.70424034130515321752707968983032e-04
            5.62616704255734512152287152275676e-01 4.33350760461963468994639470110997e-01 1.28520660558538468189182513157220e-03
            4.37383295744265598870015310239978e-01 5.57429556661395930738933657266898e-01 1.65319231840286456382271840226394e-03
            3.16084250500909769154844752847566e-01 6.77610263174368676608594341814751e-01 1.88334820460370330405530658879343e-03
            2.06341022856691147513430450999294e-01 7.86341693061848756585163755516987e-01 1.90170446945613080824699636650621e-03
            1.15048662902847709244014140494983e-01 8.76792366407857515220314326143125e-01 1.67073259312863005472682420560204e-03
            4.79413718147626011223394471016945e-02 9.43280949553400249740775507234503e-01 1.20075982838622834746045153764271e-03
            9.21968287664043373297317884862423e-03 9.81645636799064913446954960818402e-01 5.51248461238496674161058130891888e-04
            9.90780317123359566267026821151376e-01 8.77767863183721505637713278247247e-03 1.16280914861644382420546514622650e-05
            9.52058628185237454388811784156132e-01 4.56429966832812908372751792285271e-02 1.37064615065405021130523488537278e-04
            8.84951337097152235244834628247190e-01 1.09533072177831009019755015287956e-01 4.92369982151638543171889850214029e-04
            7.93658977143308685953115855227225e-01 1.96448751159280204925394741621858e-01 1.12077158116436923981917850312584e-03
            6.83915749499090064311701553378953e-01 3.00930737922855240373820606691879e-01 1.97311914805246542761740791149805e-03
            5.62616704255734512152287152275676e-01 4.16414540537423327748456358676776e-01 2.91336827023298394062633676071528e-03
            4.37383295744265598870015310239978e-01 5.35644087647813904595750500448048e-01 3.74753601801949961722093540572587e-03
            3.16084250500909769154844752847566e-01 6.51127890262382269526142408722080e-01 4.26926447253476830379081974342625e-03
            2.06341022856691147513430450999294e-01 7.55609877025957388241295120678842e-01 4.31087533832758516838534745829747e-03
            1.15048662902847709244014140494983e-01 8.42525556007406417613481153239263e-01 3.78729715806899345717506655262241e-03
            4.79413718147626011223394471016945e-02 9.06415631501956253757157355721574e-01 2.72194024601784450093089340327879e-03
            9.21968287664043373297317884862423e-03 9.43280949553400249740775507234503e-01 1.24959657770782966122735668790256e-03
            9.90780317123359566267026821151376e-01 8.15897068929467145226297475346655e-03 1.74061827295957629788396986425525e-05
            9.52058628185237454388811784156132e-01 4.24257810897458939591864179874392e-02 2.05173113612739396825063176166282e-04
            8.84951337097152235244834628247190e-01 1.01812468067114622605195961568825e-01 7.37032546578815939836148984909414e-04
            7.93658977143308685953115855227225e-01 1.82601764075022987565688481481629e-01 1.67769190353350672838261825603468e-03
            6.83915749499090064311701553378953e-01 2.79719180116131449498340089121484e-01 2.95357776287965504563581831121155e-03
            5.62616704255734512152287152275676e-01 3.87062932392846892337701092401403e-01 4.36104416022346785819507886117208e-03
            4.37383295744265598870015310239978e-01 4.97888404704305231884831073330133e-01 5.60971650360703351367863689347359e-03
            3.16084250500909769154844752847566e-01 6.05232156981020841257645770383533e-01 6.39069598122188071293026112584812e-03
            2.06341022856691147513430450999294e-01 7.02349573022129414212599840539042e-01 6.45298361753676892604847381562649e-03
            1.15048662902847709244014140494983e-01 7.83138869030037598761850858863909e-01 5.66923527072905105578248807773889e-03
            4.79413718147626011223394471016945e-02 8.42525556007406417613481153239263e-01 4.07449402660792301539283144506953e-03
            9.21968287664043373297317884862423e-03 8.76792366407857515220314326143125e-01 1.87053106657613103964987733007774e-03
            9.90780317123359566267026821151376e-01 7.31728408146012452434314710103536e-03 2.20914934983585914787026366346012e-05
            9.52058628185237454388811784156132e-01 3.80491001173515336342134673941473e-02 2.60400604533874389934222648435025e-04
            8.84951337097152235244834628247190e-01 9.13094041211794382739697084616637e-02 9.35423347196052403992871937532527e-04
            7.93658977143308685953115855227225e-01 1.63764405143145663767967334933928e-01 2.12928476937918748079781039450609e-03
            6.83915749499090064311701553378953e-01 2.50863102943661542276743148249807e-01 3.74860731724995172378656071998648e-03
            5.62616704255734512152287152275676e-01 3.47133179119962997916104541218374e-01 5.53492860601888323618391396507832e-03
            4.37383295744265598870015310239978e-01 4.46525798023345577014708851493197e-01 7.11971243737182825361831106647514e-03
            3.16084250500909769154844752847566e-01 5.42795874199647254698675169493072e-01 8.11091213463483416690191774023333e-03
            2.06341022856691147513430450999294e-01 6.29894572000163188718602214066777e-01 8.18996605093888496718079750280594e-03
            1.15048662902847709244014140494983e-01 7.02349573022129192167994915507734e-01 7.19525217387423279141733800656766e-03
            4.79413718147626011223394471016945e-02 7.55609877025957166196690195647534e-01 5.17124631495812468739980971577097e-03
            9.21968287664043373297317884862423e-03 7.86341693061848534540558830485679e-01 2.37403143111229600684120022435764e-03
            9.90780317123359566267026821151376e-01 6.30548632472146923511058247413530e-03 2.53889067556255797512235888868659e-05
            9.52058628185237454388811784156132e-01 3.27878592367079127467555110797548e-02 2.99268434164955447144423894556553e-04
            8.84951337097152235244834628247190e-01 7.86835925180692508096313986243331e-02 1.07504620005706452191829303188797e-03
            7.93658977143308685953115855227225e-01 1.41119875299442865124177615143708e-01 2.44710537429072491621973028941284e-03
            6.83915749499090064311701553378953e-01 2.16174997086187953287961249770888e-01 4.30813071321711239802798587561483e-03
            5.62616704255734512152287152275676e-01 2.99133324527321442509730786696309e-01 6.36108130433550144483545452089857e-03
            4.37383295744265598870015310239978e-01 3.84782424971768510779668304166989e-01 8.18241262016675130475107380334521e-03
            3.16084250500909769154844752847566e-01 4.67740752412902194290467150494806e-01 9.32156044156165139447089984514605e-03
            2.06341022856691147513430450999294e-01 5.42795874199647254698675169493072e-01 9.41241407759398594767841217390014e-03
            1.15048662902847709244014140494983e-01 6.05232156981020730235343307867879e-01 8.26922754404442288123089355167394e-03
            4.79413718147626011223394471016945e-02 6.51127890262382158503839946206426e-01 5.94311518642229048153824422229263e-03
            9.21968287664043373297317884862423e-03 6.77610263174368565586291879299097e-01 2.72838333197856575465523043533267e-03
            9.90780317123359566267026821151376e-01 5.18714759433847039105103249312378e-03 2.70911062442331601152027686918444e-05
            9.52058628185237454388811784156132e-01 2.69726166079204965342341893119738e-02 3.19332889105659184503238634533773e-04
            8.84951337097152235244834628247190e-01 6.47282995514291692451536164298886e-02 1.14712268249801013254851600464690e-03
            7.93658977143308685953115855227225e-01 1.16090906232388824115275838266825e-01 2.61117157677750837130314920386809e-03
            6.83915749499090064311701553378953e-01 1.77834279283965945861467616850859e-01 4.59696937679083574013194635199397e-03
            5.62616704255734512152287152275676e-01 2.46079148348149839842236019649135e-01 6.78756005930717526419826413075498e-03
            4.37383295744265598870015310239978e-01 3.16537555907584589043324285739800e-01 8.73100255007926845385313185943232e-03
            3.16084250500909769154844752847566e-01 3.84782424971768621801970766682643e-01 9.94652454771147859213797914890165e-03
            2.06341022856691147513430450999294e-01 4.46525798023345799059313776524505e-01 1.00434694666131762302851981871754e-02
            1.15048662902847709244014140494983e-01 4.97888404704305342907133535845787e-01 8.82363798133248104782477838625709e-03
            4.79413718147626011223394471016945e-02 5.35644087647814015618052962963702e-01 6.34157139914686557302658798107586e-03
            9.21968287664043373297317884862423e-03 5.57429556661396041761236119782552e-01 2.91130781774399981579048990454339e-03
            9.90780317123359566267026821151376e-01 4.03253528230196420928388434390399e-03 2.70911062442331330101484565542336e-05
            9.52058628185237454388811784156132e-01 2.09687552068421115269991616969492e-02 3.19332889105658859242586888882443e-04
            8.84951337097152235244834628247190e-01 5.03203633514185608155422357867792e-02 1.14712268249800904834634351914247e-03
            7.93658977143308685953115855227225e-01 9.02501166243024621760326908770367e-02 2.61117157677750576921793523865745e-03
            6.83915749499090064311701553378953e-01 1.38249971216944017582406445399101e-01 4.59696937679083053596151842157269e-03
            5.62616704255734512152287152275676e-01 1.91304147396115620249901212446275e-01 6.78756005930716832530436022352660e-03
            4.37383295744265598870015310239978e-01 2.46079148348149895353387250906962e-01 8.73100255007925978023575197539685e-03
            3.16084250500909769154844752847566e-01 2.99133324527321609043184480469790e-01 9.94652454771146818379712328805908e-03
            2.06341022856691147513430450999294e-01 3.47133179119963219960709466249682e-01 1.00434694666131675566678183031399e-02
            1.15048662902847709244014140494983e-01 3.87062932392847003360003554917057e-01 8.82363798133247237420739850222162e-03
            4.79413718147626011223394471016945e-02 4.16414540537423494281910052450257e-01 6.34157139914685950149442206225103e-03
            9.21968287664043373297317884862423e-03 4.33350760461963635528093163884478e-01 2.91130781774399678002440694513098e-03
            9.90780317123359566267026821151376e-01 2.91419655191896319681998939188361e-03 2.53889067556257051120997825233161e-05
            9.52058628185237454388811784156132e-01 1.51535125780546779672430801610972e-02 2.99268434164956910817356749987539e-04
            8.84951337097152235244834628247190e-01 3.63650703847784376177010301489645e-02 1.07504620005706972608872096230925e-03
            7.93658977143308685953115855227225e-01 6.52211475572483517781918749278702e-02 2.44710537429073662560319313286072e-03
            6.83915749499090064311701553378953e-01 9.99092534147219268891859655923326e-02 4.30813071321713321470969759729996e-03
            5.62616704255734512152287152275676e-01 1.38249971216943906560103982883447e-01 6.36108130433553180249628411502272e-03
            4.37383295744265598870015310239978e-01 1.77834279283965807083589538706292e-01 8.18241262016679120339102126990838e-03
            3.16084250500909769154844752847566e-01 2.16174997086187870021234402884147e-01 9.32156044156169649728127524213050e-03
            2.06341022856691147513430450999294e-01 2.50863102943661486765591916991980e-01 9.41241407759403278521226354769169e-03
            1.15048662902847709244014140494983e-01 2.79719180116131338476037626605830e-01 8.26922754404446277987084101823712e-03
            4.79413718147626011223394471016945e-02 3.00930737922855129351518144176225e-01 5.94311518642231910447559783960969e-03
            9.21968287664043373297317884862423e-03 3.13170053948990778636130016820971e-01 2.72838333197857919876216925558765e-03
            9.90780317123359566267026821151376e-01 1.90239879518030769074699026788267e-03 2.20914934983585067754079112045673e-05
            9.52058628185237454388811784156132e-01 9.89227169741105881450859982351176e-03 2.60400604533873414152267411481034e-04
            8.84951337097152235244834628247190e-01 2.37392587816682536228096722652481e-02 9.35423347196048826125702735367895e-04
            7.93658977143308685953115855227225e-01 4.25766177135455600732960590448783e-02 2.12928476937917924086129950467239e-03
            6.83915749499090064311701553378953e-01 6.52211475572483517781918749278702e-02 3.74860731724993741231788391132795e-03
            5.62616704255734512152287152275676e-01 9.02501166243023789093058439902961e-02 5.53492860601886241950220224339319e-03
            4.37383295744265598870015310239978e-01 1.16090906232388740848548991380085e-01 7.11971243737180049804269543756163e-03
            3.16084250500909769154844752847566e-01 1.41119875299442809613026383885881e-01 8.11091213463480294187935015770563e-03
            2.06341022856691147513430450999294e-01 1.63764405143145608256816103676101e-01 8.18996605093885374215822992027825e-03
            1.15048662902847709244014140494983e-01 1.82601764075022876543386018965975e-01 7.19525217387420503584172237765415e-03
            4.79413718147626011223394471016945e-02 1.96448751159280093903092279106204e-01 5.17124631495810473807983598248939e-03
            9.21968287664043373297317884862423e-03 2.04438624061510837437438681263302e-01 2.37403143111228689954295134612039e-03
            9.90780317123359566267026821151376e-01 1.06071218734576249755063859225857e-03 1.74061827295957290975218084705389e-05
            9.52058628185237454388811784156132e-01 5.51559072501670716315302911425533e-03 2.05173113612739017354302806239730e-04
            8.84951337097152235244834628247190e-01 1.32361948357330866388181789261580e-02 7.37032546578814530373324753753650e-04
            7.93658977143308685953115855227225e-01 2.37392587816682675005974800797048e-02 1.67769190353350347577610079952137e-03
            6.83915749499090064311701553378953e-01 3.63650703847784584343827418706496e-02 2.95357776287964940778452138658849e-03
            5.62616704255734512152287152275676e-01 5.03203633514185399988605240650941e-02 4.36104416022345918457769897713661e-03
            4.37383295744265598870015310239978e-01 6.47282995514291553673658086154319e-02 5.60971650360702310533778103263103e-03
            3.16084250500909769154844752847566e-01 7.86835925180692646874192064387898e-02 6.39069598122186856986592928819846e-03
            2.06341022856691147513430450999294e-01 9.13094041211794660295453240905772e-02 6.45298361753675678298414197797683e-03
            1.15048662902847709244014140494983e-01 1.01812468067114622605195961568825e-01 5.66923527072903978007989422849278e-03
            4.79413718147626011223394471016945e-02 1.09533072177831009019755015287956e-01 4.07449402660791520913718954943761e-03
            9.21968287664043373297317884862423e-03 1.13987950715501940024410032492597e-01 1.87053106657612735336249087936267e-03
            9.90780317123359566267026821151376e-01 4.42004244803219056066806436078309e-04 1.16280914861645026165586427890908e-05
            9.52058628185237454388811784156132e-01 2.29837513148131375451121982678160e-03 1.37064615065405780072044228390382e-04
            8.84951337097152235244834628247190e-01 5.51559072501670716315302911425533e-03 4.92369982151641362097538312525558e-04
            7.93658977143308685953115855227225e-01 9.89227169741106401867902775393304e-03 1.12077158116437552819177891905156e-03
            6.83915749499090064311701553378953e-01 1.51535125780546849061369840683255e-02 1.97311914805247626963913276654239e-03
            5.62616704255734512152287152275676e-01 2.09687552068421011186583058361066e-02 2.91336827023300042049935854038267e-03
            4.37383295744265598870015310239978e-01 2.69726166079204895953402854047454e-02 3.74753601801952043390264712741100e-03
            3.16084250500909769154844752847566e-01 3.27878592367079196856494149869832e-02 4.26926447253479172255774543032203e-03
            2.06341022856691147513430450999294e-01 3.80491001173515475120012752086041e-02 4.31087533832760945451401113359680e-03
            1.15048662902847709244014140494983e-01 4.24257810897458939591864179874392e-02 3.78729715806901427385677827430754e-03
            4.79413718147626011223394471016945e-02 4.56429966832812908372751792285271e-02 2.72194024601785967976130820034086e-03
            9.21968287664043373297317884862423e-03 4.74993675699593789762964490819286e-02 1.24959657770783681696169509223182e-03
            9.90780317123359566267026821151376e-01 8.50025523458168270982820979853045e-05 5.12962955664183502648215129404718e-06
            9.52058628185237454388811784156132e-01 4.42004244803219056066806436078309e-04 6.04648408077802278020283777149047e-05
            8.84951337097152235244834628247190e-01 1.06071218734576249755063859225857e-03 2.17204656177104083207216045359189e-04
            7.93658977143308685953115855227225e-01 1.90239879518030877494916275338710e-03 4.94418455154528916316303277511679e-04
            6.83915749499090064311701553378953e-01 2.91419655191896493154346536869070e-03 8.70424034130509683901411044359975e-04
            5.62616704255734512152287152275676e-01 4.03253528230196247456040836709690e-03 1.28520660558537644195531424173851e-03
            4.37383295744265598870015310239978e-01 5.18714759433846952368929450472024e-03 1.65319231840285393864142804432049e-03
            3.16084250500909769154844752847566e-01 6.30548632472147010247232046253885e-03 1.88334820460369116099097475114377e-03
            2.06341022856691147513430450999294e-01 7.31728408146012625906662307784245e-03 1.90170446945611844834223003175566e-03
            1.15048662902847709244014140494983e-01 8.15897068929467145226297475346655e-03 1.67073259312861921270509935055770e-03
            4.79413718147626011223394471016945e-02 8.77767863183721505637713278247247e-03 1.20075982838622075804524413911167e-03
            9.21968287664043373297317884862423e-03 9.13468032429461639087886482002432e-03 5.51248461238493096293888928727256e-04
            ];
        
    case  22
        
        % ALG. DEG.:   22
        % PTS CARD.:  144
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.3e-15
        
        xyw=[
            9.90780317123359566267026821151376e-01 9.13468032429461639087886482002432e-03 5.12962955664186890780004146606075e-06
            9.52058628185237454388811784156132e-01 4.74993675699593789762964490819286e-02 6.04648408077806208253159037102620e-05
            8.84951337097152235244834628247190e-01 1.13987950715501940024410032492597e-01 2.17204656177105492670040276514953e-04
            7.93658977143308685953115855227225e-01 2.04438624061510948459741143778956e-01 4.94418455154532168922820734024981e-04
            6.83915749499090064311701553378953e-01 3.13170053948990945169583710594452e-01 8.70424034130515321752707968983032e-04
            5.62616704255734512152287152275676e-01 4.33350760461963468994639470110997e-01 1.28520660558538468189182513157220e-03
            4.37383295744265598870015310239978e-01 5.57429556661395930738933657266898e-01 1.65319231840286456382271840226394e-03
            3.16084250500909769154844752847566e-01 6.77610263174368676608594341814751e-01 1.88334820460370330405530658879343e-03
            2.06341022856691147513430450999294e-01 7.86341693061848756585163755516987e-01 1.90170446945613080824699636650621e-03
            1.15048662902847709244014140494983e-01 8.76792366407857515220314326143125e-01 1.67073259312863005472682420560204e-03
            4.79413718147626011223394471016945e-02 9.43280949553400249740775507234503e-01 1.20075982838622834746045153764271e-03
            9.21968287664043373297317884862423e-03 9.81645636799064913446954960818402e-01 5.51248461238496674161058130891888e-04
            9.90780317123359566267026821151376e-01 8.77767863183721505637713278247247e-03 1.16280914861644382420546514622650e-05
            9.52058628185237454388811784156132e-01 4.56429966832812908372751792285271e-02 1.37064615065405021130523488537278e-04
            8.84951337097152235244834628247190e-01 1.09533072177831009019755015287956e-01 4.92369982151638543171889850214029e-04
            7.93658977143308685953115855227225e-01 1.96448751159280204925394741621858e-01 1.12077158116436923981917850312584e-03
            6.83915749499090064311701553378953e-01 3.00930737922855240373820606691879e-01 1.97311914805246542761740791149805e-03
            5.62616704255734512152287152275676e-01 4.16414540537423327748456358676776e-01 2.91336827023298394062633676071528e-03
            4.37383295744265598870015310239978e-01 5.35644087647813904595750500448048e-01 3.74753601801949961722093540572587e-03
            3.16084250500909769154844752847566e-01 6.51127890262382269526142408722080e-01 4.26926447253476830379081974342625e-03
            2.06341022856691147513430450999294e-01 7.55609877025957388241295120678842e-01 4.31087533832758516838534745829747e-03
            1.15048662902847709244014140494983e-01 8.42525556007406417613481153239263e-01 3.78729715806899345717506655262241e-03
            4.79413718147626011223394471016945e-02 9.06415631501956253757157355721574e-01 2.72194024601784450093089340327879e-03
            9.21968287664043373297317884862423e-03 9.43280949553400249740775507234503e-01 1.24959657770782966122735668790256e-03
            9.90780317123359566267026821151376e-01 8.15897068929467145226297475346655e-03 1.74061827295957629788396986425525e-05
            9.52058628185237454388811784156132e-01 4.24257810897458939591864179874392e-02 2.05173113612739396825063176166282e-04
            8.84951337097152235244834628247190e-01 1.01812468067114622605195961568825e-01 7.37032546578815939836148984909414e-04
            7.93658977143308685953115855227225e-01 1.82601764075022987565688481481629e-01 1.67769190353350672838261825603468e-03
            6.83915749499090064311701553378953e-01 2.79719180116131449498340089121484e-01 2.95357776287965504563581831121155e-03
            5.62616704255734512152287152275676e-01 3.87062932392846892337701092401403e-01 4.36104416022346785819507886117208e-03
            4.37383295744265598870015310239978e-01 4.97888404704305231884831073330133e-01 5.60971650360703351367863689347359e-03
            3.16084250500909769154844752847566e-01 6.05232156981020841257645770383533e-01 6.39069598122188071293026112584812e-03
            2.06341022856691147513430450999294e-01 7.02349573022129414212599840539042e-01 6.45298361753676892604847381562649e-03
            1.15048662902847709244014140494983e-01 7.83138869030037598761850858863909e-01 5.66923527072905105578248807773889e-03
            4.79413718147626011223394471016945e-02 8.42525556007406417613481153239263e-01 4.07449402660792301539283144506953e-03
            9.21968287664043373297317884862423e-03 8.76792366407857515220314326143125e-01 1.87053106657613103964987733007774e-03
            9.90780317123359566267026821151376e-01 7.31728408146012452434314710103536e-03 2.20914934983585914787026366346012e-05
            9.52058628185237454388811784156132e-01 3.80491001173515336342134673941473e-02 2.60400604533874389934222648435025e-04
            8.84951337097152235244834628247190e-01 9.13094041211794382739697084616637e-02 9.35423347196052403992871937532527e-04
            7.93658977143308685953115855227225e-01 1.63764405143145663767967334933928e-01 2.12928476937918748079781039450609e-03
            6.83915749499090064311701553378953e-01 2.50863102943661542276743148249807e-01 3.74860731724995172378656071998648e-03
            5.62616704255734512152287152275676e-01 3.47133179119962997916104541218374e-01 5.53492860601888323618391396507832e-03
            4.37383295744265598870015310239978e-01 4.46525798023345577014708851493197e-01 7.11971243737182825361831106647514e-03
            3.16084250500909769154844752847566e-01 5.42795874199647254698675169493072e-01 8.11091213463483416690191774023333e-03
            2.06341022856691147513430450999294e-01 6.29894572000163188718602214066777e-01 8.18996605093888496718079750280594e-03
            1.15048662902847709244014140494983e-01 7.02349573022129192167994915507734e-01 7.19525217387423279141733800656766e-03
            4.79413718147626011223394471016945e-02 7.55609877025957166196690195647534e-01 5.17124631495812468739980971577097e-03
            9.21968287664043373297317884862423e-03 7.86341693061848534540558830485679e-01 2.37403143111229600684120022435764e-03
            9.90780317123359566267026821151376e-01 6.30548632472146923511058247413530e-03 2.53889067556255797512235888868659e-05
            9.52058628185237454388811784156132e-01 3.27878592367079127467555110797548e-02 2.99268434164955447144423894556553e-04
            8.84951337097152235244834628247190e-01 7.86835925180692508096313986243331e-02 1.07504620005706452191829303188797e-03
            7.93658977143308685953115855227225e-01 1.41119875299442865124177615143708e-01 2.44710537429072491621973028941284e-03
            6.83915749499090064311701553378953e-01 2.16174997086187953287961249770888e-01 4.30813071321711239802798587561483e-03
            5.62616704255734512152287152275676e-01 2.99133324527321442509730786696309e-01 6.36108130433550144483545452089857e-03
            4.37383295744265598870015310239978e-01 3.84782424971768510779668304166989e-01 8.18241262016675130475107380334521e-03
            3.16084250500909769154844752847566e-01 4.67740752412902194290467150494806e-01 9.32156044156165139447089984514605e-03
            2.06341022856691147513430450999294e-01 5.42795874199647254698675169493072e-01 9.41241407759398594767841217390014e-03
            1.15048662902847709244014140494983e-01 6.05232156981020730235343307867879e-01 8.26922754404442288123089355167394e-03
            4.79413718147626011223394471016945e-02 6.51127890262382158503839946206426e-01 5.94311518642229048153824422229263e-03
            9.21968287664043373297317884862423e-03 6.77610263174368565586291879299097e-01 2.72838333197856575465523043533267e-03
            9.90780317123359566267026821151376e-01 5.18714759433847039105103249312378e-03 2.70911062442331601152027686918444e-05
            9.52058628185237454388811784156132e-01 2.69726166079204965342341893119738e-02 3.19332889105659184503238634533773e-04
            8.84951337097152235244834628247190e-01 6.47282995514291692451536164298886e-02 1.14712268249801013254851600464690e-03
            7.93658977143308685953115855227225e-01 1.16090906232388824115275838266825e-01 2.61117157677750837130314920386809e-03
            6.83915749499090064311701553378953e-01 1.77834279283965945861467616850859e-01 4.59696937679083574013194635199397e-03
            5.62616704255734512152287152275676e-01 2.46079148348149839842236019649135e-01 6.78756005930717526419826413075498e-03
            4.37383295744265598870015310239978e-01 3.16537555907584589043324285739800e-01 8.73100255007926845385313185943232e-03
            3.16084250500909769154844752847566e-01 3.84782424971768621801970766682643e-01 9.94652454771147859213797914890165e-03
            2.06341022856691147513430450999294e-01 4.46525798023345799059313776524505e-01 1.00434694666131762302851981871754e-02
            1.15048662902847709244014140494983e-01 4.97888404704305342907133535845787e-01 8.82363798133248104782477838625709e-03
            4.79413718147626011223394471016945e-02 5.35644087647814015618052962963702e-01 6.34157139914686557302658798107586e-03
            9.21968287664043373297317884862423e-03 5.57429556661396041761236119782552e-01 2.91130781774399981579048990454339e-03
            9.90780317123359566267026821151376e-01 4.03253528230196420928388434390399e-03 2.70911062442331330101484565542336e-05
            9.52058628185237454388811784156132e-01 2.09687552068421115269991616969492e-02 3.19332889105658859242586888882443e-04
            8.84951337097152235244834628247190e-01 5.03203633514185608155422357867792e-02 1.14712268249800904834634351914247e-03
            7.93658977143308685953115855227225e-01 9.02501166243024621760326908770367e-02 2.61117157677750576921793523865745e-03
            6.83915749499090064311701553378953e-01 1.38249971216944017582406445399101e-01 4.59696937679083053596151842157269e-03
            5.62616704255734512152287152275676e-01 1.91304147396115620249901212446275e-01 6.78756005930716832530436022352660e-03
            4.37383295744265598870015310239978e-01 2.46079148348149895353387250906962e-01 8.73100255007925978023575197539685e-03
            3.16084250500909769154844752847566e-01 2.99133324527321609043184480469790e-01 9.94652454771146818379712328805908e-03
            2.06341022856691147513430450999294e-01 3.47133179119963219960709466249682e-01 1.00434694666131675566678183031399e-02
            1.15048662902847709244014140494983e-01 3.87062932392847003360003554917057e-01 8.82363798133247237420739850222162e-03
            4.79413718147626011223394471016945e-02 4.16414540537423494281910052450257e-01 6.34157139914685950149442206225103e-03
            9.21968287664043373297317884862423e-03 4.33350760461963635528093163884478e-01 2.91130781774399678002440694513098e-03
            9.90780317123359566267026821151376e-01 2.91419655191896319681998939188361e-03 2.53889067556257051120997825233161e-05
            9.52058628185237454388811784156132e-01 1.51535125780546779672430801610972e-02 2.99268434164956910817356749987539e-04
            8.84951337097152235244834628247190e-01 3.63650703847784376177010301489645e-02 1.07504620005706972608872096230925e-03
            7.93658977143308685953115855227225e-01 6.52211475572483517781918749278702e-02 2.44710537429073662560319313286072e-03
            6.83915749499090064311701553378953e-01 9.99092534147219268891859655923326e-02 4.30813071321713321470969759729996e-03
            5.62616704255734512152287152275676e-01 1.38249971216943906560103982883447e-01 6.36108130433553180249628411502272e-03
            4.37383295744265598870015310239978e-01 1.77834279283965807083589538706292e-01 8.18241262016679120339102126990838e-03
            3.16084250500909769154844752847566e-01 2.16174997086187870021234402884147e-01 9.32156044156169649728127524213050e-03
            2.06341022856691147513430450999294e-01 2.50863102943661486765591916991980e-01 9.41241407759403278521226354769169e-03
            1.15048662902847709244014140494983e-01 2.79719180116131338476037626605830e-01 8.26922754404446277987084101823712e-03
            4.79413718147626011223394471016945e-02 3.00930737922855129351518144176225e-01 5.94311518642231910447559783960969e-03
            9.21968287664043373297317884862423e-03 3.13170053948990778636130016820971e-01 2.72838333197857919876216925558765e-03
            9.90780317123359566267026821151376e-01 1.90239879518030769074699026788267e-03 2.20914934983585067754079112045673e-05
            9.52058628185237454388811784156132e-01 9.89227169741105881450859982351176e-03 2.60400604533873414152267411481034e-04
            8.84951337097152235244834628247190e-01 2.37392587816682536228096722652481e-02 9.35423347196048826125702735367895e-04
            7.93658977143308685953115855227225e-01 4.25766177135455600732960590448783e-02 2.12928476937917924086129950467239e-03
            6.83915749499090064311701553378953e-01 6.52211475572483517781918749278702e-02 3.74860731724993741231788391132795e-03
            5.62616704255734512152287152275676e-01 9.02501166243023789093058439902961e-02 5.53492860601886241950220224339319e-03
            4.37383295744265598870015310239978e-01 1.16090906232388740848548991380085e-01 7.11971243737180049804269543756163e-03
            3.16084250500909769154844752847566e-01 1.41119875299442809613026383885881e-01 8.11091213463480294187935015770563e-03
            2.06341022856691147513430450999294e-01 1.63764405143145608256816103676101e-01 8.18996605093885374215822992027825e-03
            1.15048662902847709244014140494983e-01 1.82601764075022876543386018965975e-01 7.19525217387420503584172237765415e-03
            4.79413718147626011223394471016945e-02 1.96448751159280093903092279106204e-01 5.17124631495810473807983598248939e-03
            9.21968287664043373297317884862423e-03 2.04438624061510837437438681263302e-01 2.37403143111228689954295134612039e-03
            9.90780317123359566267026821151376e-01 1.06071218734576249755063859225857e-03 1.74061827295957290975218084705389e-05
            9.52058628185237454388811784156132e-01 5.51559072501670716315302911425533e-03 2.05173113612739017354302806239730e-04
            8.84951337097152235244834628247190e-01 1.32361948357330866388181789261580e-02 7.37032546578814530373324753753650e-04
            7.93658977143308685953115855227225e-01 2.37392587816682675005974800797048e-02 1.67769190353350347577610079952137e-03
            6.83915749499090064311701553378953e-01 3.63650703847784584343827418706496e-02 2.95357776287964940778452138658849e-03
            5.62616704255734512152287152275676e-01 5.03203633514185399988605240650941e-02 4.36104416022345918457769897713661e-03
            4.37383295744265598870015310239978e-01 6.47282995514291553673658086154319e-02 5.60971650360702310533778103263103e-03
            3.16084250500909769154844752847566e-01 7.86835925180692646874192064387898e-02 6.39069598122186856986592928819846e-03
            2.06341022856691147513430450999294e-01 9.13094041211794660295453240905772e-02 6.45298361753675678298414197797683e-03
            1.15048662902847709244014140494983e-01 1.01812468067114622605195961568825e-01 5.66923527072903978007989422849278e-03
            4.79413718147626011223394471016945e-02 1.09533072177831009019755015287956e-01 4.07449402660791520913718954943761e-03
            9.21968287664043373297317884862423e-03 1.13987950715501940024410032492597e-01 1.87053106657612735336249087936267e-03
            9.90780317123359566267026821151376e-01 4.42004244803219056066806436078309e-04 1.16280914861645026165586427890908e-05
            9.52058628185237454388811784156132e-01 2.29837513148131375451121982678160e-03 1.37064615065405780072044228390382e-04
            8.84951337097152235244834628247190e-01 5.51559072501670716315302911425533e-03 4.92369982151641362097538312525558e-04
            7.93658977143308685953115855227225e-01 9.89227169741106401867902775393304e-03 1.12077158116437552819177891905156e-03
            6.83915749499090064311701553378953e-01 1.51535125780546849061369840683255e-02 1.97311914805247626963913276654239e-03
            5.62616704255734512152287152275676e-01 2.09687552068421011186583058361066e-02 2.91336827023300042049935854038267e-03
            4.37383295744265598870015310239978e-01 2.69726166079204895953402854047454e-02 3.74753601801952043390264712741100e-03
            3.16084250500909769154844752847566e-01 3.27878592367079196856494149869832e-02 4.26926447253479172255774543032203e-03
            2.06341022856691147513430450999294e-01 3.80491001173515475120012752086041e-02 4.31087533832760945451401113359680e-03
            1.15048662902847709244014140494983e-01 4.24257810897458939591864179874392e-02 3.78729715806901427385677827430754e-03
            4.79413718147626011223394471016945e-02 4.56429966832812908372751792285271e-02 2.72194024601785967976130820034086e-03
            9.21968287664043373297317884862423e-03 4.74993675699593789762964490819286e-02 1.24959657770783681696169509223182e-03
            9.90780317123359566267026821151376e-01 8.50025523458168270982820979853045e-05 5.12962955664183502648215129404718e-06
            9.52058628185237454388811784156132e-01 4.42004244803219056066806436078309e-04 6.04648408077802278020283777149047e-05
            8.84951337097152235244834628247190e-01 1.06071218734576249755063859225857e-03 2.17204656177104083207216045359189e-04
            7.93658977143308685953115855227225e-01 1.90239879518030877494916275338710e-03 4.94418455154528916316303277511679e-04
            6.83915749499090064311701553378953e-01 2.91419655191896493154346536869070e-03 8.70424034130509683901411044359975e-04
            5.62616704255734512152287152275676e-01 4.03253528230196247456040836709690e-03 1.28520660558537644195531424173851e-03
            4.37383295744265598870015310239978e-01 5.18714759433846952368929450472024e-03 1.65319231840285393864142804432049e-03
            3.16084250500909769154844752847566e-01 6.30548632472147010247232046253885e-03 1.88334820460369116099097475114377e-03
            2.06341022856691147513430450999294e-01 7.31728408146012625906662307784245e-03 1.90170446945611844834223003175566e-03
            1.15048662902847709244014140494983e-01 8.15897068929467145226297475346655e-03 1.67073259312861921270509935055770e-03
            4.79413718147626011223394471016945e-02 8.77767863183721505637713278247247e-03 1.20075982838622075804524413911167e-03
            9.21968287664043373297317884862423e-03 9.13468032429461639087886482002432e-03 5.51248461238493096293888928727256e-04
            ];
        
    case  23
        
        % ALG. DEG.:   23
        % PTS CARD.:  169
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.1e-15
        
        xyw=[
            9.92091527359294067522910154366400e-01 7.84592870119713882426371043266045e-03 3.24040698608366363001075972916087e-06
            9.58799199611488850436558095680084e-01 4.08749649858634253596356700199976e-02 3.84140528073924396529684188728027e-05
            9.00789045366654939073214336531237e-01 9.84263475129689285880729698874347e-02 1.39444862231932135542694650709450e-04
            8.21174669720170169462392095738323e-01 1.77411095047846606442831784988812e-01 3.22424875105556082207497459179990e-04
            7.24246375518223350908897373301443e-01 2.73572834486986993240975607477594e-01 5.79994381036458045604320421517741e-04
            6.15229157977567386872408405906754e-01 3.81727892345356867309646986541338e-01 8.81206929324765517348716503676087e-04
            5.00000000000000000000000000000000e-01 4.96045763679647033761455077183200e-01 1.17682727364753591907386987003292e-03
            3.84770842022432724149894056608900e-01 6.10363635013937089190960705309408e-01 1.40900540769371392449405000490970e-03
            2.75753624481776538068800164182903e-01 7.18518692872307185304237009404460e-01 1.52331208366165873728526047869991e-03
            1.78825330279829886048759135519504e-01 8.14680432311447377813351522490848e-01 1.48059080841656518813531651090898e-03
            9.92109546333450609267856634687632e-02 8.93665179846325097301473761035595e-01 1.26609410014656828188939208246211e-03
            4.12008003885109275188369792886078e-02 9.51216562373430885024561121099396e-01 8.93947756797264692259219476255794e-04
            7.90847264070587696593861437577289e-03 9.84245598658096909616688208188862e-01 4.06498253473444527868185804919676e-04
            9.92091527359294067522910154366400e-01 7.58263723805820567119839736847098e-03 7.37355786249749108247387996417643e-06
            9.58799199611488850436558095680084e-01 3.95032944358572057907963426259812e-02 8.74113166416402696421925067049585e-05
            9.00789045366654939073214336531237e-01 9.51233838951429794272840467783681e-02 3.17307290322140910388687418119957e-04
            8.21174669720170169462392095738323e-01 1.71457583542560992873049485751835e-01 7.33678974002175657460889102168267e-04
            7.24246375518223350908897373301443e-01 2.64392354443094501625921566301258e-01 1.31977931996265715475225022146333e-03
            6.15229157977567386872408405906754e-01 3.68917975364947081651223470544210e-01 2.00518956727188538483042101745468e-03
            5.00000000000000000000000000000000e-01 4.79399599805744425218279047840042e-01 2.67787473415267898019243020257818e-03
            3.84770842022432724149894056608900e-01 5.89881224246541768785334625135874e-01 3.20619692119538703123371448100443e-03
            2.75753624481776538068800164182903e-01 6.94406845168394459832938991894480e-01 3.46630217739903826595915248276469e-03
            1.78825330279829886048759135519504e-01 7.87341616068927718785630531783681e-01 3.36908976046122179623387715707850e-03
            9.92109546333450609267856634687632e-02 8.63675815716345884887061856716173e-01 2.88100172197208650309852018267520e-03
            4.12008003885109275188369792886078e-02 9.19295905175631866690366678085411e-01 2.03418136644650457411920285721862e-03
            7.90847264070587696593861437577289e-03 9.51216562373430662979956196068088e-01 9.24988251742161473809011873470354e-04
            9.92091527359294067522910154366400e-01 7.12386552032980534254757998269270e-03 1.11156664294165642633020404939082e-05
            9.58799199611488850436558095680084e-01 3.71132296503090611250463837222924e-02 1.31772891196310567045110739847757e-04
            9.00789045366654939073214336531237e-01 8.93681411140854059516058782719483e-02 4.78341943009897619960152370666151e-04
            8.21174669720170169462392095738323e-01 1.61083898550144682948470631345117e-01 1.10602383454037043498952019859871e-03
            7.24246375518223350908897373301443e-01 2.48395844153334627302243120539060e-01 1.98957232786645425368088346829154e-03
            6.15229157977567386872408405906754e-01 3.46597359470311117135565837088507e-01 3.02283087393999654740595062207831e-03
            5.00000000000000000000000000000000e-01 4.50394522683327469536607168265618e-01 4.03690631302940691055303190637460e-03
            3.84770842022432724149894056608900e-01 5.54191685896343821937648499442730e-01 4.83335401276135858544824586147115e-03
            2.75753624481776538068800164182903e-01 6.52393201213320450548849294136744e-01 5.22546367249601857729590648204976e-03
            1.78825330279829886048759135519504e-01 7.39705146816510117346865627041552e-01 5.07891558544918634426901249412367e-03
            9.92109546333450609267856634687632e-02 8.11420904252569519243820650444832e-01 4.34312101718145777973223431445149e-03
            4.12008003885109275188369792886078e-02 8.63675815716346106931666781747481e-01 3.06653612109791980089301688394698e-03
            7.90847264070587696593861437577289e-03 8.93665179846325097301473761035595e-01 1.39442329594908782236517552632904e-03
            9.92091527359294067522910154366400e-01 6.49423740872269623924939452308536e-03 1.42591002038983847243196895204775e-05
            9.58799199611488850436558095680084e-01 3.38330536512423049599895819028461e-02 1.69037355668850736330199868717727e-04
            9.00789045366654939073214336531237e-01 8.14695229036599133376483905522036e-02 6.13613744206570658167032572549715e-04
            8.21174669720170169462392095738323e-01 1.46846831530139609878204964843462e-01 1.41879974401488474257393423982876e-03
            7.24246375518223350908897373301443e-01 2.26441891507962761442840360359696e-01 2.55220965527302133535325623370227e-03
            6.15229157977567386872408405906754e-01 3.15964069115722945291224732500268e-01 3.87766658928163834157976097571918e-03
            5.00000000000000000000000000000000e-01 4.10587334860085084731196047869162e-01 5.17851556600350693831957826773760e-03
            3.84770842022432724149894056608900e-01 5.05210600604447224171167363238055e-01 6.20019318018487624460943763438081e-03
            2.75753624481776538068800164182903e-01 5.94732778212207491286278582265368e-01 6.70318874635953462076676245828821e-03
            1.78825330279829886048759135519504e-01 6.74327838190030504073035899637034e-01 6.51519787139405708725758969990238e-03
            9.92109546333450609267856634687632e-02 7.39705146816510228369168089557206e-01 5.57132567578298536214997938031956e-03
            4.12008003885109275188369792886078e-02 7.87341616068928051852537919330643e-01 3.93373137879454975313064934994145e-03
            7.90847264070587696593861437577289e-03 8.14680432311447488835653985006502e-01 1.78875658331822777469266583239005e-03
            9.92091527359294067522910154366400e-01 5.72768264591630438253844204155030e-03 1.66339416310849862320423581119044e-05
            9.58799199611488850436558095680084e-01 2.98395303498290090371547478298453e-02 1.97190388415945891962058333923835e-04
            9.00789045366654939073214336531237e-01 7.18531743049030530157139651237230e-02 7.15810609309913171982575619267664e-04
            8.21174669720170169462392095738323e-01 1.29513597306015942889345637922816e-01 1.65509967604335187819908359330157e-03
            7.24246375518223350908897373301443e-01 1.99713563066939964185664280194032e-01 2.97727807709047406423130688324363e-03
            6.15229157977567386872408405906754e-01 2.78668887739841775186278027831577e-01 4.52348877478852932909614281697941e-03
            5.00000000000000000000000000000000e-01 3.62123187759111675454448686650721e-01 6.04099308012557775771078638626932e-03
            3.84770842022432724149894056608900e-01 4.45577487778381520211468114212039e-01 7.23283026178959606578011687361141e-03
            2.75753624481776538068800164182903e-01 5.24532812451283469989959939994151e-01 7.81959932637309174108253273516311e-03
            1.78825330279829886048759135519504e-01 5.94732778212207269241673657234060e-01 7.60029872558922077463439137545720e-03
            9.92109546333450609267856634687632e-02 6.52393201213320339526546831621090e-01 6.49922539105257223573941516292507e-03
            4.12008003885109275188369792886078e-02 6.94406845168394459832938991894480e-01 4.58889110894576159049851327154101e-03
            7.90847264070587696593861437577289e-03 7.18518692872307074281934546888806e-01 2.08667252306704142233995113997480e-03
            9.92091527359294067522910154366400e-01 4.86552296363013982161138670790024e-03 1.81120816087178764875256775646051e-05
            9.58799199611488850436558095680084e-01 2.53479337310255452286877897449813e-02 2.14713294458724604160670956787271e-04
            9.00789045366654939073214336531237e-01 6.10374720812235163625025791134249e-02 7.79419602385700836279236458636888e-04
            8.21174669720170169462392095738323e-01 1.10018557373120093312302003596415e-01 1.80217660178866312392642612394411e-03
            7.24246375518223350908897373301443e-01 1.69651670199185755638637829179061e-01 3.24184758490053334667457818341063e-03
            6.15229157977567386872408405906754e-01 2.36722241151780865564546729729045e-01 4.92545902000649394453235530022539e-03
            5.00000000000000000000000000000000e-01 3.07614578988783693436204202953377e-01 6.57781313001983870825117861613762e-03
            3.84770842022432724149894056608900e-01 3.78506916825786521307861676177708e-01 7.87556039746621118491365365343881e-03
            2.75753624481776538068800164182903e-01 4.45577487778381686744921807985520e-01 8.51447145167760939543644127525113e-03
            1.78825330279829886048759135519504e-01 5.05210600604447224171167363238055e-01 8.27568316767788615673229202229777e-03
            9.92109546333450609267856634687632e-02 5.54191685896343821937648499442730e-01 7.07676528431567825744297195456056e-03
            4.12008003885109275188369792886078e-02 5.89881224246541990829939550167182e-01 4.99667319400856884542561076045786e-03
            7.90847264070587696593861437577289e-03 6.10363635013937200213263167825062e-01 2.27210025540977534827535855299629e-03
            9.92091527359294067522910154366400e-01 3.95423632035296623854492281679995e-03 1.86138125929561861924763088715906e-05
            9.58799199611488850436558095680084e-01 2.06004001942555747817209521599580e-02 2.20661164774523785086710048197745e-04
            9.00789045366654939073214336531237e-01 4.96054773166725304633928317343816e-02 8.01010658162049603292498289874857e-04
            8.21174669720170169462392095738323e-01 8.94126651399149152688039521308383e-02 1.85209951289963406853567029486385e-03
            7.24246375518223350908897373301443e-01 1.37876812240888324545551313349279e-01 3.33165147462791957563843503464796e-03
            6.15229157977567386872408405906754e-01 1.92385421011216334319371412675537e-01 5.06190139340789366229067525182472e-03
            5.00000000000000000000000000000000e-01 2.50000000000000000000000000000000e-01 6.76002811376151969274816977417686e-03
            3.84770842022432724149894056608900e-01 3.07614578988783637925052971695550e-01 8.09372486663175599863606635153701e-03
            2.75753624481776538068800164182903e-01 3.62123187759111730965599917908548e-01 8.75033468562322856720836483646053e-03
            1.78825330279829886048759135519504e-01 4.10587334860085029220044816611335e-01 8.50493161910739238484868707246278e-03
            9.92109546333450609267856634687632e-02 4.50394522683327469536607168265618e-01 7.27280196789677375879268339531336e-03
            4.12008003885109275188369792886078e-02 4.79399599805744536240581510355696e-01 5.13508830353084488179460365131490e-03
            7.90847264070587696593861437577289e-03 4.96045763679647033761455077183200e-01 2.33504073470212512517174552328925e-03
            9.92091527359294067522910154366400e-01 3.04294967707579352284019691410322e-03 1.81120816087178155011534752549807e-05
            9.58799199611488850436558095680084e-01 1.58528666574856078042010665285488e-02 2.14713294458723899429258841209389e-04
            9.00789045366654939073214336531237e-01 3.81734825521215515031769882625667e-02 7.79419602385698234194022493426246e-04
            8.21174669720170169462392095738323e-01 6.88067729067097649808815162941755e-02 1.80217660178865705239426020511928e-03
            7.24246375518223350908897373301443e-01 1.06101954282590921208040413148410e-01 3.24184758490052250465285332836629e-03
            6.15229157977567386872408405906754e-01 1.48048600870651858585347326879855e-01 4.92545902000647746465933352055799e-03
            5.00000000000000000000000000000000e-01 1.92385421011216362074947028304450e-01 6.57781313001981702420772890604894e-03
            3.84770842022432724149894056608900e-01 2.36722241151780837808971114100132e-01 7.87556039746618516406151400133240e-03
            2.75753624481776538068800164182903e-01 2.78668887739841830697429259089404e-01 8.51447145167757990513734966953052e-03
            1.78825330279829886048759135519504e-01 3.15964069115722945291224732500268e-01 8.27568316767785840115667639338426e-03
            9.92109546333450609267856634687632e-02 3.46597359470311172646717068346334e-01 7.07676528431565483867604626766479e-03
            4.12008003885109275188369792886078e-02 3.68917975364947192673525933059864e-01 4.99667319400855236555258898079046e-03
            7.90847264070587696593861437577289e-03 3.81727892345356922820798217799165e-01 2.27210025540976754201971665736437e-03
            9.92091527359294067522910154366400e-01 2.18078999478962722718966560364606e-03 1.66339416310849692913834130258977e-05
            9.58799199611488850436558095680084e-01 1.13612700386821370568402045364564e-02 1.97190388415945702226678148960559e-04
            9.00789045366654939073214336531237e-01 2.73577803284420009721777944378118e-02 7.15810609309912521461272127965003e-04
            8.21174669720170169462392095738323e-01 4.93117329738138598926866507099476e-02 1.65509967604335036031604211359536e-03
            7.24246375518223350908897373301443e-01 7.60400614148366571498627308756113e-02 2.97727807709047102846522392383122e-03
            6.15229157977567386872408405906754e-01 1.06101954282590865696889181890583e-01 4.52348877478852499228745287496167e-03
            5.00000000000000000000000000000000e-01 1.37876812240888269034400082091452e-01 6.04099308012557255354035845584804e-03
            3.84770842022432724149894056608900e-01 1.69651670199185672371910982292320e-01 7.23283026178958912688621296638303e-03
            2.75753624481776538068800164182903e-01 1.99713563066939908674513048936205e-01 7.81959932637308480218862882793474e-03
            1.78825330279829886048759135519504e-01 2.26441891507962650420537897844042e-01 7.60029872558921383574048746822882e-03
            9.92109546333450609267856634687632e-02 2.48395844153334544035516273652320e-01 6.49922539105256616420724924410024e-03
            4.12008003885109275188369792886078e-02 2.64392354443094446114770335043431e-01 4.58889110894575812105156131792683e-03
            7.90847264070587696593861437577289e-03 2.73572834486986937729824376219767e-01 2.08667252306703925393560616896593e-03
            9.92091527359294067522910154366400e-01 1.41423523198323688836175460181721e-03 1.42591002038984101353081071494877e-05
            9.58799199611488850436558095680084e-01 7.36774673726884980762275034749109e-03 1.69037355668851034485797302231447e-04
            9.00789045366654939073214336531237e-01 1.77414317296851510585842248701738e-02 6.13613744206571742369205058054149e-04
            8.21174669720170169462392095738323e-01 3.19784987496902345371907472326711e-02 1.41879974401488712781871370793851e-03
            7.24246375518223350908897373301443e-01 4.93117329738138945871561702460895e-02 2.55220965527302567216194617572000e-03
            6.15229157977567386872408405906754e-01 6.88067729067097649808815162941755e-02 3.87766658928164484679279588874579e-03
            5.00000000000000000000000000000000e-01 8.94126651399149430243795677597518e-02 5.17851556600351561193695815177307e-03
            3.84770842022432724149894056608900e-01 1.10018557373120107190089811410871e-01 6.20019318018488752031203148362692e-03
            2.75753624481776538068800164182903e-01 1.29513597306016026156072484809556e-01 6.70318874635954589646935630753433e-03
            1.78825330279829886048759135519504e-01 1.46846831530139637633780580472376e-01 6.51519787139406836296018354914850e-03
            9.92109546333450609267856634687632e-02 1.61083898550144738459621862602944e-01 5.57132567578299490312909725275858e-03
            4.12008003885109275188369792886078e-02 1.71457583542561076139776332638576e-01 3.93373137879455669202455325716983e-03
            7.90847264070587696593861437577289e-03 1.77411095047846661953983016246639e-01 1.78875658331823081045874879180246e-03
            9.92091527359294067522910154366400e-01 7.84607120376126917701831153806324e-04 1.11156664294165778158291965627136e-05
            9.58799199611488850436558095680084e-01 4.08757073820208496894856864400936e-03 1.31772891196310729675436612673423e-04
            9.00789045366654939073214336531237e-01 9.84281351925965150573283324320073e-03 4.78341943009898162061238613418368e-04
            8.21174669720170169462392095738323e-01 1.77414317296851441196903209629454e-02 1.10602383454037173603212718120403e-03
            7.24246375518223350908897373301443e-01 2.73577803284420113805186502986544e-02 1.98957232786645685576609743350218e-03
            6.15229157977567386872408405906754e-01 3.81734825521215445642830843553384e-02 3.02283087394000001685290257569250e-03
            5.00000000000000000000000000000000e-01 4.96054773166725304633928317343816e-02 4.03690631302941211472345983679588e-03
            3.84770842022432724149894056608900e-01 6.10374720812235094236086752061965e-02 4.83335401276136378961867379189243e-03
            2.75753624481776538068800164182903e-01 7.18531743049030530157139651237230e-02 5.22546367249602464882807240087459e-03
            1.78825330279829886048759135519504e-01 8.14695229036598994598605827377469e-02 5.07891558544919241580117841294850e-03
            9.92109546333450609267856634687632e-02 8.93681411140854059516058782719483e-02 4.34312101718146298390266224487277e-03
            4.12008003885109275188369792886078e-02 9.51233838951429933050718545928248e-02 3.06653612109792327033996883756117e-03
            7.90847264070587696593861437577289e-03 9.84263475129689285880729698874347e-02 1.39442329594908955708865150313613e-03
            9.92091527359294067522910154366400e-01 3.25835402647725016957863664046613e-04 7.37355786249750463500103603298186e-06
            9.58799199611488850436558095680084e-01 1.69750595265393141274079535918418e-03 8.74113166416404322725183795306236e-05
            9.00789045366654939073214336531237e-01 4.08757073820206328490511893392068e-03 3.17307290322141452489773660872174e-04
            8.21174669720170169462392095738323e-01 7.36774673726880817425932690412083e-03 7.33678974002176958503496084773587e-04
            7.24246375518223350908897373301443e-01 1.13612700386820798109654973018223e-02 1.31977931996265953999702968957308e-03
            6.15229157977567386872408405906754e-01 1.58528666574855210680272676881941e-02 2.00518956727188885427737297106887e-03
            5.00000000000000000000000000000000e-01 2.06004001942554637594184896443039e-02 2.67787473415268418436285813299946e-03
            3.84770842022432724149894056608900e-01 2.53479337310254029813627596467995e-02 3.20619692119539310276588039982926e-03
            2.75753624481776538068800164182903e-01 2.98395303498288529120419099172068e-02 3.46630217739904477117218739579130e-03
            1.78825330279829886048759135519504e-01 3.38330536512421176098541764076799e-02 3.36908976046122786776604307590333e-03
            9.92109546333450609267856634687632e-02 3.71132296503088668360170743198978e-02 2.88100172197209170726894811309649e-03
            4.12008003885109275188369792886078e-02 3.95032944358570045628731293163582e-02 2.03418136644650804356615481083281e-03
            7.90847264070587696593861437577289e-03 4.08749649858632033150307449886895e-02 9.24988251742163208532487850277448e-04
            9.92091527359294067522910154366400e-01 6.25439395087938290089882298339319e-05 3.24040698608367633550496854366596e-06
            9.58799199611488850436558095680084e-01 3.25835402647724474856777421294396e-04 3.84140528073925955070307136640650e-05
            9.00789045366654939073214336531237e-01 7.84607120376121388270751477733711e-04 1.39444862231932704748835205599278e-04
            8.21174669720170169462392095738323e-01 1.41423523198322648002089874097464e-03 3.22424875105557329039995817510089e-04
            7.24246375518223350908897373301443e-01 2.18078999478961291572098879498753e-03 5.79994381036460322428882641077053e-04
            6.15229157977567386872408405906754e-01 3.04294967707577183879674720401454e-03 8.81206929324768986795668457290276e-04
            5.00000000000000000000000000000000e-01 3.95423632035293848296930718788644e-03 1.17682727364754068956342880625243e-03
            3.84770842022432724149894056608900e-01 4.86552296363010425978012918335480e-03 1.40900540769371934550491243243187e-03
            2.75753624481776538068800164182903e-01 5.72768264591626448389849457498713e-03 1.52331208366166480881742639752474e-03
            1.78825330279829886048759135519504e-01 6.49423740872264940171554314929381e-03 1.48059080841657104282704793263292e-03
            9.92109546333450609267856634687632e-02 7.12386552032975590292851464369051e-03 1.26609410014657326921938551578251e-03
            4.12008003885109275188369792886078e-02 7.58263723805815449685585605266169e-03 8.93947756797268270126388678420426e-04
            7.90847264070587696593861437577289e-03 7.84592870119708331311247917483342e-03 4.06498253473446099961335908901106e-04
            ];
        
    case  24
        
        % ALG. DEG.:   24
        % PTS CARD.:  169
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.1e-15
        
        xyw=[
            9.92091527359294067522910154366400e-01 7.84592870119713882426371043266045e-03 3.24040698608366363001075972916087e-06
            9.58799199611488850436558095680084e-01 4.08749649858634253596356700199976e-02 3.84140528073924396529684188728027e-05
            9.00789045366654939073214336531237e-01 9.84263475129689285880729698874347e-02 1.39444862231932135542694650709450e-04
            8.21174669720170169462392095738323e-01 1.77411095047846606442831784988812e-01 3.22424875105556082207497459179990e-04
            7.24246375518223350908897373301443e-01 2.73572834486986993240975607477594e-01 5.79994381036458045604320421517741e-04
            6.15229157977567386872408405906754e-01 3.81727892345356867309646986541338e-01 8.81206929324765517348716503676087e-04
            5.00000000000000000000000000000000e-01 4.96045763679647033761455077183200e-01 1.17682727364753591907386987003292e-03
            3.84770842022432724149894056608900e-01 6.10363635013937089190960705309408e-01 1.40900540769371392449405000490970e-03
            2.75753624481776538068800164182903e-01 7.18518692872307185304237009404460e-01 1.52331208366165873728526047869991e-03
            1.78825330279829886048759135519504e-01 8.14680432311447377813351522490848e-01 1.48059080841656518813531651090898e-03
            9.92109546333450609267856634687632e-02 8.93665179846325097301473761035595e-01 1.26609410014656828188939208246211e-03
            4.12008003885109275188369792886078e-02 9.51216562373430885024561121099396e-01 8.93947756797264692259219476255794e-04
            7.90847264070587696593861437577289e-03 9.84245598658096909616688208188862e-01 4.06498253473444527868185804919676e-04
            9.92091527359294067522910154366400e-01 7.58263723805820567119839736847098e-03 7.37355786249749108247387996417643e-06
            9.58799199611488850436558095680084e-01 3.95032944358572057907963426259812e-02 8.74113166416402696421925067049585e-05
            9.00789045366654939073214336531237e-01 9.51233838951429794272840467783681e-02 3.17307290322140910388687418119957e-04
            8.21174669720170169462392095738323e-01 1.71457583542560992873049485751835e-01 7.33678974002175657460889102168267e-04
            7.24246375518223350908897373301443e-01 2.64392354443094501625921566301258e-01 1.31977931996265715475225022146333e-03
            6.15229157977567386872408405906754e-01 3.68917975364947081651223470544210e-01 2.00518956727188538483042101745468e-03
            5.00000000000000000000000000000000e-01 4.79399599805744425218279047840042e-01 2.67787473415267898019243020257818e-03
            3.84770842022432724149894056608900e-01 5.89881224246541768785334625135874e-01 3.20619692119538703123371448100443e-03
            2.75753624481776538068800164182903e-01 6.94406845168394459832938991894480e-01 3.46630217739903826595915248276469e-03
            1.78825330279829886048759135519504e-01 7.87341616068927718785630531783681e-01 3.36908976046122179623387715707850e-03
            9.92109546333450609267856634687632e-02 8.63675815716345884887061856716173e-01 2.88100172197208650309852018267520e-03
            4.12008003885109275188369792886078e-02 9.19295905175631866690366678085411e-01 2.03418136644650457411920285721862e-03
            7.90847264070587696593861437577289e-03 9.51216562373430662979956196068088e-01 9.24988251742161473809011873470354e-04
            9.92091527359294067522910154366400e-01 7.12386552032980534254757998269270e-03 1.11156664294165642633020404939082e-05
            9.58799199611488850436558095680084e-01 3.71132296503090611250463837222924e-02 1.31772891196310567045110739847757e-04
            9.00789045366654939073214336531237e-01 8.93681411140854059516058782719483e-02 4.78341943009897619960152370666151e-04
            8.21174669720170169462392095738323e-01 1.61083898550144682948470631345117e-01 1.10602383454037043498952019859871e-03
            7.24246375518223350908897373301443e-01 2.48395844153334627302243120539060e-01 1.98957232786645425368088346829154e-03
            6.15229157977567386872408405906754e-01 3.46597359470311117135565837088507e-01 3.02283087393999654740595062207831e-03
            5.00000000000000000000000000000000e-01 4.50394522683327469536607168265618e-01 4.03690631302940691055303190637460e-03
            3.84770842022432724149894056608900e-01 5.54191685896343821937648499442730e-01 4.83335401276135858544824586147115e-03
            2.75753624481776538068800164182903e-01 6.52393201213320450548849294136744e-01 5.22546367249601857729590648204976e-03
            1.78825330279829886048759135519504e-01 7.39705146816510117346865627041552e-01 5.07891558544918634426901249412367e-03
            9.92109546333450609267856634687632e-02 8.11420904252569519243820650444832e-01 4.34312101718145777973223431445149e-03
            4.12008003885109275188369792886078e-02 8.63675815716346106931666781747481e-01 3.06653612109791980089301688394698e-03
            7.90847264070587696593861437577289e-03 8.93665179846325097301473761035595e-01 1.39442329594908782236517552632904e-03
            9.92091527359294067522910154366400e-01 6.49423740872269623924939452308536e-03 1.42591002038983847243196895204775e-05
            9.58799199611488850436558095680084e-01 3.38330536512423049599895819028461e-02 1.69037355668850736330199868717727e-04
            9.00789045366654939073214336531237e-01 8.14695229036599133376483905522036e-02 6.13613744206570658167032572549715e-04
            8.21174669720170169462392095738323e-01 1.46846831530139609878204964843462e-01 1.41879974401488474257393423982876e-03
            7.24246375518223350908897373301443e-01 2.26441891507962761442840360359696e-01 2.55220965527302133535325623370227e-03
            6.15229157977567386872408405906754e-01 3.15964069115722945291224732500268e-01 3.87766658928163834157976097571918e-03
            5.00000000000000000000000000000000e-01 4.10587334860085084731196047869162e-01 5.17851556600350693831957826773760e-03
            3.84770842022432724149894056608900e-01 5.05210600604447224171167363238055e-01 6.20019318018487624460943763438081e-03
            2.75753624481776538068800164182903e-01 5.94732778212207491286278582265368e-01 6.70318874635953462076676245828821e-03
            1.78825330279829886048759135519504e-01 6.74327838190030504073035899637034e-01 6.51519787139405708725758969990238e-03
            9.92109546333450609267856634687632e-02 7.39705146816510228369168089557206e-01 5.57132567578298536214997938031956e-03
            4.12008003885109275188369792886078e-02 7.87341616068928051852537919330643e-01 3.93373137879454975313064934994145e-03
            7.90847264070587696593861437577289e-03 8.14680432311447488835653985006502e-01 1.78875658331822777469266583239005e-03
            9.92091527359294067522910154366400e-01 5.72768264591630438253844204155030e-03 1.66339416310849862320423581119044e-05
            9.58799199611488850436558095680084e-01 2.98395303498290090371547478298453e-02 1.97190388415945891962058333923835e-04
            9.00789045366654939073214336531237e-01 7.18531743049030530157139651237230e-02 7.15810609309913171982575619267664e-04
            8.21174669720170169462392095738323e-01 1.29513597306015942889345637922816e-01 1.65509967604335187819908359330157e-03
            7.24246375518223350908897373301443e-01 1.99713563066939964185664280194032e-01 2.97727807709047406423130688324363e-03
            6.15229157977567386872408405906754e-01 2.78668887739841775186278027831577e-01 4.52348877478852932909614281697941e-03
            5.00000000000000000000000000000000e-01 3.62123187759111675454448686650721e-01 6.04099308012557775771078638626932e-03
            3.84770842022432724149894056608900e-01 4.45577487778381520211468114212039e-01 7.23283026178959606578011687361141e-03
            2.75753624481776538068800164182903e-01 5.24532812451283469989959939994151e-01 7.81959932637309174108253273516311e-03
            1.78825330279829886048759135519504e-01 5.94732778212207269241673657234060e-01 7.60029872558922077463439137545720e-03
            9.92109546333450609267856634687632e-02 6.52393201213320339526546831621090e-01 6.49922539105257223573941516292507e-03
            4.12008003885109275188369792886078e-02 6.94406845168394459832938991894480e-01 4.58889110894576159049851327154101e-03
            7.90847264070587696593861437577289e-03 7.18518692872307074281934546888806e-01 2.08667252306704142233995113997480e-03
            9.92091527359294067522910154366400e-01 4.86552296363013982161138670790024e-03 1.81120816087178764875256775646051e-05
            9.58799199611488850436558095680084e-01 2.53479337310255452286877897449813e-02 2.14713294458724604160670956787271e-04
            9.00789045366654939073214336531237e-01 6.10374720812235163625025791134249e-02 7.79419602385700836279236458636888e-04
            8.21174669720170169462392095738323e-01 1.10018557373120093312302003596415e-01 1.80217660178866312392642612394411e-03
            7.24246375518223350908897373301443e-01 1.69651670199185755638637829179061e-01 3.24184758490053334667457818341063e-03
            6.15229157977567386872408405906754e-01 2.36722241151780865564546729729045e-01 4.92545902000649394453235530022539e-03
            5.00000000000000000000000000000000e-01 3.07614578988783693436204202953377e-01 6.57781313001983870825117861613762e-03
            3.84770842022432724149894056608900e-01 3.78506916825786521307861676177708e-01 7.87556039746621118491365365343881e-03
            2.75753624481776538068800164182903e-01 4.45577487778381686744921807985520e-01 8.51447145167760939543644127525113e-03
            1.78825330279829886048759135519504e-01 5.05210600604447224171167363238055e-01 8.27568316767788615673229202229777e-03
            9.92109546333450609267856634687632e-02 5.54191685896343821937648499442730e-01 7.07676528431567825744297195456056e-03
            4.12008003885109275188369792886078e-02 5.89881224246541990829939550167182e-01 4.99667319400856884542561076045786e-03
            7.90847264070587696593861437577289e-03 6.10363635013937200213263167825062e-01 2.27210025540977534827535855299629e-03
            9.92091527359294067522910154366400e-01 3.95423632035296623854492281679995e-03 1.86138125929561861924763088715906e-05
            9.58799199611488850436558095680084e-01 2.06004001942555747817209521599580e-02 2.20661164774523785086710048197745e-04
            9.00789045366654939073214336531237e-01 4.96054773166725304633928317343816e-02 8.01010658162049603292498289874857e-04
            8.21174669720170169462392095738323e-01 8.94126651399149152688039521308383e-02 1.85209951289963406853567029486385e-03
            7.24246375518223350908897373301443e-01 1.37876812240888324545551313349279e-01 3.33165147462791957563843503464796e-03
            6.15229157977567386872408405906754e-01 1.92385421011216334319371412675537e-01 5.06190139340789366229067525182472e-03
            5.00000000000000000000000000000000e-01 2.50000000000000000000000000000000e-01 6.76002811376151969274816977417686e-03
            3.84770842022432724149894056608900e-01 3.07614578988783637925052971695550e-01 8.09372486663175599863606635153701e-03
            2.75753624481776538068800164182903e-01 3.62123187759111730965599917908548e-01 8.75033468562322856720836483646053e-03
            1.78825330279829886048759135519504e-01 4.10587334860085029220044816611335e-01 8.50493161910739238484868707246278e-03
            9.92109546333450609267856634687632e-02 4.50394522683327469536607168265618e-01 7.27280196789677375879268339531336e-03
            4.12008003885109275188369792886078e-02 4.79399599805744536240581510355696e-01 5.13508830353084488179460365131490e-03
            7.90847264070587696593861437577289e-03 4.96045763679647033761455077183200e-01 2.33504073470212512517174552328925e-03
            9.92091527359294067522910154366400e-01 3.04294967707579352284019691410322e-03 1.81120816087178155011534752549807e-05
            9.58799199611488850436558095680084e-01 1.58528666574856078042010665285488e-02 2.14713294458723899429258841209389e-04
            9.00789045366654939073214336531237e-01 3.81734825521215515031769882625667e-02 7.79419602385698234194022493426246e-04
            8.21174669720170169462392095738323e-01 6.88067729067097649808815162941755e-02 1.80217660178865705239426020511928e-03
            7.24246375518223350908897373301443e-01 1.06101954282590921208040413148410e-01 3.24184758490052250465285332836629e-03
            6.15229157977567386872408405906754e-01 1.48048600870651858585347326879855e-01 4.92545902000647746465933352055799e-03
            5.00000000000000000000000000000000e-01 1.92385421011216362074947028304450e-01 6.57781313001981702420772890604894e-03
            3.84770842022432724149894056608900e-01 2.36722241151780837808971114100132e-01 7.87556039746618516406151400133240e-03
            2.75753624481776538068800164182903e-01 2.78668887739841830697429259089404e-01 8.51447145167757990513734966953052e-03
            1.78825330279829886048759135519504e-01 3.15964069115722945291224732500268e-01 8.27568316767785840115667639338426e-03
            9.92109546333450609267856634687632e-02 3.46597359470311172646717068346334e-01 7.07676528431565483867604626766479e-03
            4.12008003885109275188369792886078e-02 3.68917975364947192673525933059864e-01 4.99667319400855236555258898079046e-03
            7.90847264070587696593861437577289e-03 3.81727892345356922820798217799165e-01 2.27210025540976754201971665736437e-03
            9.92091527359294067522910154366400e-01 2.18078999478962722718966560364606e-03 1.66339416310849692913834130258977e-05
            9.58799199611488850436558095680084e-01 1.13612700386821370568402045364564e-02 1.97190388415945702226678148960559e-04
            9.00789045366654939073214336531237e-01 2.73577803284420009721777944378118e-02 7.15810609309912521461272127965003e-04
            8.21174669720170169462392095738323e-01 4.93117329738138598926866507099476e-02 1.65509967604335036031604211359536e-03
            7.24246375518223350908897373301443e-01 7.60400614148366571498627308756113e-02 2.97727807709047102846522392383122e-03
            6.15229157977567386872408405906754e-01 1.06101954282590865696889181890583e-01 4.52348877478852499228745287496167e-03
            5.00000000000000000000000000000000e-01 1.37876812240888269034400082091452e-01 6.04099308012557255354035845584804e-03
            3.84770842022432724149894056608900e-01 1.69651670199185672371910982292320e-01 7.23283026178958912688621296638303e-03
            2.75753624481776538068800164182903e-01 1.99713563066939908674513048936205e-01 7.81959932637308480218862882793474e-03
            1.78825330279829886048759135519504e-01 2.26441891507962650420537897844042e-01 7.60029872558921383574048746822882e-03
            9.92109546333450609267856634687632e-02 2.48395844153334544035516273652320e-01 6.49922539105256616420724924410024e-03
            4.12008003885109275188369792886078e-02 2.64392354443094446114770335043431e-01 4.58889110894575812105156131792683e-03
            7.90847264070587696593861437577289e-03 2.73572834486986937729824376219767e-01 2.08667252306703925393560616896593e-03
            9.92091527359294067522910154366400e-01 1.41423523198323688836175460181721e-03 1.42591002038984101353081071494877e-05
            9.58799199611488850436558095680084e-01 7.36774673726884980762275034749109e-03 1.69037355668851034485797302231447e-04
            9.00789045366654939073214336531237e-01 1.77414317296851510585842248701738e-02 6.13613744206571742369205058054149e-04
            8.21174669720170169462392095738323e-01 3.19784987496902345371907472326711e-02 1.41879974401488712781871370793851e-03
            7.24246375518223350908897373301443e-01 4.93117329738138945871561702460895e-02 2.55220965527302567216194617572000e-03
            6.15229157977567386872408405906754e-01 6.88067729067097649808815162941755e-02 3.87766658928164484679279588874579e-03
            5.00000000000000000000000000000000e-01 8.94126651399149430243795677597518e-02 5.17851556600351561193695815177307e-03
            3.84770842022432724149894056608900e-01 1.10018557373120107190089811410871e-01 6.20019318018488752031203148362692e-03
            2.75753624481776538068800164182903e-01 1.29513597306016026156072484809556e-01 6.70318874635954589646935630753433e-03
            1.78825330279829886048759135519504e-01 1.46846831530139637633780580472376e-01 6.51519787139406836296018354914850e-03
            9.92109546333450609267856634687632e-02 1.61083898550144738459621862602944e-01 5.57132567578299490312909725275858e-03
            4.12008003885109275188369792886078e-02 1.71457583542561076139776332638576e-01 3.93373137879455669202455325716983e-03
            7.90847264070587696593861437577289e-03 1.77411095047846661953983016246639e-01 1.78875658331823081045874879180246e-03
            9.92091527359294067522910154366400e-01 7.84607120376126917701831153806324e-04 1.11156664294165778158291965627136e-05
            9.58799199611488850436558095680084e-01 4.08757073820208496894856864400936e-03 1.31772891196310729675436612673423e-04
            9.00789045366654939073214336531237e-01 9.84281351925965150573283324320073e-03 4.78341943009898162061238613418368e-04
            8.21174669720170169462392095738323e-01 1.77414317296851441196903209629454e-02 1.10602383454037173603212718120403e-03
            7.24246375518223350908897373301443e-01 2.73577803284420113805186502986544e-02 1.98957232786645685576609743350218e-03
            6.15229157977567386872408405906754e-01 3.81734825521215445642830843553384e-02 3.02283087394000001685290257569250e-03
            5.00000000000000000000000000000000e-01 4.96054773166725304633928317343816e-02 4.03690631302941211472345983679588e-03
            3.84770842022432724149894056608900e-01 6.10374720812235094236086752061965e-02 4.83335401276136378961867379189243e-03
            2.75753624481776538068800164182903e-01 7.18531743049030530157139651237230e-02 5.22546367249602464882807240087459e-03
            1.78825330279829886048759135519504e-01 8.14695229036598994598605827377469e-02 5.07891558544919241580117841294850e-03
            9.92109546333450609267856634687632e-02 8.93681411140854059516058782719483e-02 4.34312101718146298390266224487277e-03
            4.12008003885109275188369792886078e-02 9.51233838951429933050718545928248e-02 3.06653612109792327033996883756117e-03
            7.90847264070587696593861437577289e-03 9.84263475129689285880729698874347e-02 1.39442329594908955708865150313613e-03
            9.92091527359294067522910154366400e-01 3.25835402647725016957863664046613e-04 7.37355786249750463500103603298186e-06
            9.58799199611488850436558095680084e-01 1.69750595265393141274079535918418e-03 8.74113166416404322725183795306236e-05
            9.00789045366654939073214336531237e-01 4.08757073820206328490511893392068e-03 3.17307290322141452489773660872174e-04
            8.21174669720170169462392095738323e-01 7.36774673726880817425932690412083e-03 7.33678974002176958503496084773587e-04
            7.24246375518223350908897373301443e-01 1.13612700386820798109654973018223e-02 1.31977931996265953999702968957308e-03
            6.15229157977567386872408405906754e-01 1.58528666574855210680272676881941e-02 2.00518956727188885427737297106887e-03
            5.00000000000000000000000000000000e-01 2.06004001942554637594184896443039e-02 2.67787473415268418436285813299946e-03
            3.84770842022432724149894056608900e-01 2.53479337310254029813627596467995e-02 3.20619692119539310276588039982926e-03
            2.75753624481776538068800164182903e-01 2.98395303498288529120419099172068e-02 3.46630217739904477117218739579130e-03
            1.78825330279829886048759135519504e-01 3.38330536512421176098541764076799e-02 3.36908976046122786776604307590333e-03
            9.92109546333450609267856634687632e-02 3.71132296503088668360170743198978e-02 2.88100172197209170726894811309649e-03
            4.12008003885109275188369792886078e-02 3.95032944358570045628731293163582e-02 2.03418136644650804356615481083281e-03
            7.90847264070587696593861437577289e-03 4.08749649858632033150307449886895e-02 9.24988251742163208532487850277448e-04
            9.92091527359294067522910154366400e-01 6.25439395087938290089882298339319e-05 3.24040698608367633550496854366596e-06
            9.58799199611488850436558095680084e-01 3.25835402647724474856777421294396e-04 3.84140528073925955070307136640650e-05
            9.00789045366654939073214336531237e-01 7.84607120376121388270751477733711e-04 1.39444862231932704748835205599278e-04
            8.21174669720170169462392095738323e-01 1.41423523198322648002089874097464e-03 3.22424875105557329039995817510089e-04
            7.24246375518223350908897373301443e-01 2.18078999478961291572098879498753e-03 5.79994381036460322428882641077053e-04
            6.15229157977567386872408405906754e-01 3.04294967707577183879674720401454e-03 8.81206929324768986795668457290276e-04
            5.00000000000000000000000000000000e-01 3.95423632035293848296930718788644e-03 1.17682727364754068956342880625243e-03
            3.84770842022432724149894056608900e-01 4.86552296363010425978012918335480e-03 1.40900540769371934550491243243187e-03
            2.75753624481776538068800164182903e-01 5.72768264591626448389849457498713e-03 1.52331208366166480881742639752474e-03
            1.78825330279829886048759135519504e-01 6.49423740872264940171554314929381e-03 1.48059080841657104282704793263292e-03
            9.92109546333450609267856634687632e-02 7.12386552032975590292851464369051e-03 1.26609410014657326921938551578251e-03
            4.12008003885109275188369792886078e-02 7.58263723805815449685585605266169e-03 8.93947756797268270126388678420426e-04
            7.90847264070587696593861437577289e-03 7.84592870119708331311247917483342e-03 4.06498253473446099961335908901106e-04
            ];
        
    case  25
        
        % ALG. DEG.:   25
        % PTS CARD.:  196
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.4e-15
        
        xyw=[
            9.93141904348405990532455689390190e-01 6.81106217562754169547734051093357e-03 2.11465349255880850790154795726394e-06
            9.64217441831786703509976632631151e-01 3.55371579616369642762307989869441e-02 2.51829482589999107951077295108888e-05
            9.13600657534882620858240898087388e-01 8.58068075102567801026864913183090e-02 9.21808975185356356844487701884816e-05
            8.43646452405842683930359271471389e-01 1.55281260009290517087876537516422e-01 2.15802699707498936779193066115567e-04
            7.57624318179077160806400570436381e-01 2.40713446211374693106677113974001e-01 3.94830494084022216427082296874573e-04
            6.59556184463944927820477914792718e-01 3.38109019285115164343835658655735e-01 6.13348625727785229377886366819439e-04
            5.54027474353671722795411369588692e-01 4.42914003407462730788068938636570e-01 8.42882539237887616796751633074791e-04
            4.45972525646328055159983705380000e-01 5.50227900940943426277840444527101e-01 1.04710505140199789005095443172877e-03
            3.40443815536054961157219622691628e-01 6.55032885063290937210922493250109e-01 1.18826620097135359638285123651258e-03
            2.42375681820922894704750660821446e-01 7.52428458137031408448081037931843e-01 1.23417160347682021773185745416868e-03
            1.56353547594157371580791959786438e-01 8.37860644339115445689003536244854e-01 1.16441989855200567563486835354070e-03
            8.63993424651176011863640269439202e-02 9.07335096838148946751800849597203e-01 9.74735758192728880475175223807582e-04
            3.57825581682132409788721361110220e-02 9.57604746386769178911890776362270e-01 6.78594242310091074615874262576654e-04
            6.85809565159406497869554186763708e-03 9.86330842172778443632807920948835e-01 3.06229470005820977093063950391638e-04
            9.93141904348405990532455689390190e-01 6.61269544501762260996224895848172e-03 4.82657129033283007678257689465617e-06
            9.64217441831786703509976632631151e-01 3.45021666991517303446634912234003e-02 5.74785871541296319474009801631098e-05
            9.13600657534882620858240898087388e-01 8.33077529676639338962473857463920e-02 2.10397436291895838624335657307540e-04
            8.43646452405842683930359271471389e-01 1.50758817682562884510133471849258e-01 4.92556874423985481710464018334505e-04
            7.57624318179077160806400570436381e-01 2.33702859887605363109486233952339e-01 9.01177206572944481527409621435254e-04
            6.59556184463944927820477914792718e-01 3.28261864903627686995690737603582e-01 1.39993189348514338564422043731383e-03
            5.54027474353671722795411369588692e-01 4.30014487805963463440406258087023e-01 1.92382944975335516517300771255350e-03
            4.45972525646328055159983705380000e-01 5.34202954025823406603024068317609e-01 2.38995523230802614511603110258875e-03
            3.40443815536054961157219622691628e-01 6.35955576928159072025437126285397e-01 2.71214719151993601670147526760957e-03
            2.42375681820922894704750660821446e-01 7.30514581944181395911641629936639e-01 2.81692355255672936126254413125025e-03
            1.56353547594157371580791959786438e-01 8.13458624149223874510994392039720e-01 2.65771941929019393791033820662051e-03
            8.63993424651176011863640269439202e-02 8.80909688864122575324699937482364e-01 2.22477660889068475785768086439020e-03
            3.57825581682132409788721361110220e-02 9.29715275132635077248721700016176e-01 1.54885114712379292657706830738107e-03
            6.85809565159406497869554186763708e-03 9.57604746386769067889588313846616e-01 6.98950619278742050696773269180540e-04
            9.93141904348405990532455689390190e-01 6.26556069673335578468220674608347e-03 7.31701647714661541639213221399629e-06
            9.64217441831786703509976632631151e-01 3.26909686707598512445116512026289e-02 8.71367569214693101420018961356107e-05
            9.13600657534882620858240898087388e-01 7.89344960867127409809285154551617e-02 3.18959653860421434284216912757870e-04
            8.43646452405842683930359271471389e-01 1.42844703889933699603176364689716e-01 7.46709527177337288594838327071557e-04
            7.57624318179077160806400570436381e-01 2.21434582282060643532162202973268e-01 1.36617239706584561384494769242792e-03
            6.59556184463944927820477914792718e-01 3.11029693727424227933653355648858e-01 2.12227772373947427136942067704695e-03
            5.54027474353671722795411369588692e-01 4.07440792672977780686238702401170e-01 2.91649929863451937150919945906935e-03
            4.45972525646328055159983705380000e-01 5.06159864861905006705455889459699e-01 3.62313964976879133647602593271131e-03
            3.40443815536054961157219622691628e-01 6.02570963807458559458041236212011e-01 4.11157828095186621969459039860340e-03
            2.42375681820922894704750660821446e-01 6.92166075252822032837229926371947e-01 4.27041782024494779440582092888690e-03
            1.56353547594157371580791959786438e-01 7.70755953644948976766215764655499e-01 4.02906651799146784537208887400084e-03
            8.63993424651176011863640269439202e-02 8.34666161448169630077131841972005e-01 3.37273110164730766449836529830009e-03
            3.57825581682132409788721361110220e-02 8.80909688864122908391607325029327e-01 2.34803728826115222852455310942332e-03
            6.85809565159406497869554186763708e-03 9.07335096838149168796405774628511e-01 1.05959963923411143449371252245328e-03
            9.93141904348405990532455689390190e-01 5.78580806672717492394886207307536e-03 9.46569859935521720035404152904235e-06
            9.64217441831786703509976632631151e-01 3.01878282566188545221752548286531e-02 1.12724944725771028660940564769533e-04
            9.13600657534882620858240898087388e-01 7.28904987608937487975069302592601e-02 4.12623910063243497237106538477747e-04
            8.43646452405842683930359271471389e-01 1.31907115748878894789797300290957e-01 9.65984885725451342813885347027281e-04
            7.57624318179077160806400570436381e-01 2.04479384117668888665164672602259e-01 1.76735643356469270329189846790996e-03
            6.59556184463944927820477914792718e-01 2.87214217220501943916843856641208e-01 2.74549624697271055923097904383212e-03
            5.54027474353671722795411369588692e-01 3.76243139131998471924589466652833e-01 3.77294535448017848838775378794708e-03
            4.45972525646328055159983705380000e-01 4.67403313273844323028072267334210e-01 4.68709453029124249734405083245292e-03
            3.40443815536054961157219622691628e-01 5.56432235185340795524666646088008e-01 5.31896585127309556861874284550140e-03
            2.42375681820922894704750660821446e-01 6.39167068288173823020770214498043e-01 5.52444949468222819277674062732331e-03
            1.56353547594157371580791959786438e-01 7.11739336656963761384986355551518e-01 5.21222405541627541170068838027873e-03
            8.63993424651176011863640269439202e-02 7.70755953644948754721610839624191e-01 4.36315213510554342213332290612016e-03
            3.57825581682132409788721361110220e-02 8.13458624149223874510994392039720e-01 3.03755134898844912985649280301459e-03
            6.85809565159406497869554186763708e-03 8.37860644339115445689003536244854e-01 1.37075689966013555300128334835108e-03
            9.93141904348405990532455689390190e-01 5.19586004204576363402212635378419e-03 1.11718522017194268669331921883803e-05
            9.64217441831786703509976632631151e-01 2.71097362348957683642058924533558e-02 1.33043156688835388775857260235114e-04
            9.13600657534882620858240898087388e-01 6.54582429262551418469584518788906e-02 4.86997688520964878197000436088615e-04
            8.43646452405842683930359271471389e-01 1.18457249890903323907842548123881e-01 1.14009972524951180838925779426063e-03
            7.57624318179077160806400570436381e-01 1.83629710682765651119652261513693e-01 2.08591522921379186189527743522376e-03
            6.59556184463944927820477914792718e-01 2.57928513623787214470439721480943e-01 3.24036075833260012873937583322004e-03
            5.54027474353671722795411369588692e-01 3.37879630669400410347691376955481e-01 4.45300338089020530940587150325882e-03
            4.45972525646328055159983705380000e-01 4.19744687509676861481011655996554e-01 5.53192422072980392183660924843025e-03
            3.40443815536054961157219622691628e-01 4.99695804555290001847112080213265e-01 6.27768777261337312944888822130451e-03
            2.42375681820922894704750660821446e-01 5.73994607496311592953475155809429e-01 6.52020900545655958391311202149154e-03
            1.56353547594157371580791959786438e-01 6.39167068288173823020770214498043e-01 6.15170620299739824565765644592830e-03
            8.63993424651176011863640269439202e-02 6.92166075252821810792625001340639e-01 5.14959252878979811818282996682683e-03
            3.57825581682132409788721361110220e-02 7.30514581944181506933944092452293e-01 3.58505760244088103605442086063704e-03
            6.85809565159406497869554186763708e-03 7.52428458137031408448081037931843e-01 1.61783024535910435054431033563560e-03
            9.93141904348405990532455689390190e-01 4.52329940065408081517617233657802e-03 1.23556468088325947275729621566143e-05
            9.64217441831786703509976632631151e-01 2.36006075357859286534267795332198e-02 1.47140709051487992209059085801925e-04
            9.13600657534882620858240898087388e-01 5.69852206564865071403147567252745e-02 5.38601059827556959656857937801533e-04
            8.43646452405842683930359271471389e-01 1.03123949278604215562538115591451e-01 1.26090725849935703269111275659498e-03
            7.57624318179077160806400570436381e-01 1.59860379908655036995668297095108e-01 2.30694350229267023180690365791179e-03
            6.59556184463944927820477914792718e-01 2.24541823999307582626627777244721e-01 3.58371667833169314440477926098083e-03
            5.54027474353671722795411369588692e-01 2.94143937391041054851825720106717e-01 4.92485364283183884442918198942607e-03
            4.45972525646328055159983705380000e-01 3.65412247072903983990954657201655e-01 6.11809936350968773532965627737212e-03
            3.40443815536054961157219622691628e-01 4.35014360464637428460576984434738e-01 6.94288570006323848410323407165379e-03
            2.42375681820922894704750660821446e-01 4.99695804555289946335960848955438e-01 7.21110502865334335870217685737771e-03
            1.56353547594157371580791959786438e-01 5.56432235185340684502364183572354e-01 6.80355483974647186340423132833166e-03
            8.63993424651176011863640269439202e-02 6.02570963807458226391133848665049e-01 5.69525494486377691777390097627176e-03
            3.57825581682132409788721361110220e-02 6.35955576928159072025437126285397e-01 3.96493837595365567622041069739680e-03
            6.85809565159406497869554186763708e-03 6.55032885063290826188620030734455e-01 1.78925917989029349165763793649830e-03
            9.93141904348405990532455689390190e-01 3.79957341272849687813240038281037e-03 1.29617156769339195052049770739977e-05
            9.64217441831786703509976632631151e-01 1.98245203278485590536739380240761e-02 1.54358251311008734529828245740646e-04
            9.13600657534882620858240898087388e-01 4.78676094917669173178786934386153e-02 5.65020505101317170940089518182958e-04
            8.43646452405842683930359271471389e-01 8.66241610798275801785806038424198e-02 1.32275724877207643236098899564013e-03
            7.57624318179077160806400570436381e-01 1.34282786843995055603429022994533e-01 2.42010363537519977358836165137745e-03
            6.59556184463944927820477914792718e-01 1.88615227280767905337910406160518e-01 3.75950505626425606625629072254924e-03
            5.54027474353671722795411369588692e-01 2.47081032014963314358269030890369e-01 5.16642743650341448868879012934485e-03
            4.45972525646328055159983705380000e-01 3.06946442338708491703869185585063e-01 6.41820421544856464096673320796071e-03
            3.40443815536054961157219622691628e-01 3.65412247072903872968652194686001e-01 7.28344795007722888013912410087869e-03
            2.42375681820922894704750660821446e-01 4.19744687509676694947557962223073e-01 7.56482396624481422814367803653113e-03
            1.56353547594157371580791959786438e-01 4.67403313273844156494618573560729e-01 7.13728263599933653027473923202706e-03
            8.63993424651176011863640269439202e-02 5.06159864861904673638548501912737e-01 5.97461844330196122782039580556557e-03
            3.57825581682132409788721361110220e-02 5.34202954025823184558419143286301e-01 4.15942643075042396694218638231177e-03
            6.85809565159406497869554186763708e-03 5.50227900940943204233235519495793e-01 1.87702587496293093159704934436149e-03
            9.93141904348405990532455689390190e-01 3.05852223886545534353720299236556e-03 1.29617156769339381399298166686052e-05
            9.64217441831786703509976632631151e-01 1.59580378403647304974555254375446e-02 1.54358251311008951370262742841533e-04
            9.13600657534882620858240898087388e-01 3.85317329733504410071986967523117e-02 5.65020505101317929881610258036062e-04
            8.43646452405842683930359271471389e-01 6.97293865143296942576967012428213e-02 1.32275724877207816708446497244722e-03
            7.57624318179077160806400570436381e-01 1.08092894976927783590170406569086e-01 2.42010363537520324303531360499164e-03
            6.59556184463944927820477914792718e-01 1.51828588255287111330460447788937e-01 3.75950505626426170410758764717229e-03
            5.54027474353671722795411369588692e-01 1.98891493631364796312865905747458e-01 5.16642743650342229494443202497678e-03
            4.45972525646328055159983705380000e-01 2.47081032014963314358269030890369e-01 6.41820421544857418194585108039973e-03
            3.40443815536054961157219622691628e-01 2.94143937391040999340674488848890e-01 7.28344795007723928847997996172126e-03
            2.42375681820922894704750660821446e-01 3.37879630669400299325388914439827e-01 7.56482396624482463648453389737369e-03
            1.56353547594157371580791959786438e-01 3.76243139131998360902287004137179e-01 7.13728263599934607125385710446608e-03
            8.63993424651176011863640269439202e-02 4.07440792672977503130482546112034e-01 5.97461844330196990143777568960104e-03
            3.57825581682132409788721361110220e-02 4.30014487805963352418103795571369e-01 4.15942643075043003847435230113661e-03
            6.85809565159406497869554186763708e-03 4.42914003407462564254615244863089e-01 1.87702587496293353368226330957214e-03
            9.93141904348405990532455689390190e-01 2.33479625093987270753603802120324e-03 1.23556468088325388233984433727919e-05
            9.64217441831786703509976632631151e-01 1.21819506324273643671496358820150e-02 1.47140709051487341687755594499265e-04
            9.13600657534882620858240898087388e-01 2.94141218086308615931034893264950e-02 5.38601059827554574412078469691778e-04
            8.43646452405842683930359271471389e-01 5.32295983155530866293148051227035e-02 1.26090725849935139483981583197192e-03
            7.57624318179077160806400570436381e-01 8.25153019122678299535067480974249e-02 2.30694350229266025714691679127100e-03
            6.59556184463944927820477914792718e-01 1.15901991536747461797318692333647e-01 3.58371667833167709821262647551521e-03
            5.54027474353671722795411369588692e-01 1.51828588255287111330460447788937e-01 4.92485364283181716038573227933739e-03
            4.45972525646328055159983705380000e-01 1.88615227280767905337910406160518e-01 6.11809936350966084711577863686216e-03
            3.40443815536054961157219622691628e-01 2.24541823999307554871052161615808e-01 6.94288570006320725908066648912609e-03
            2.42375681820922894704750660821446e-01 2.57928513623787158959288490223116e-01 7.21110502865331126631787128644646e-03
            1.56353547594157371580791959786438e-01 2.87214217220501888405692625383381e-01 6.80355483974644150574340173420751e-03
            8.63993424651176011863640269439202e-02 3.11029693727424005889048430617549e-01 5.69525494486375089692176132416535e-03
            3.57825581682132409788721361110220e-02 3.28261864903627631484539506345754e-01 3.96493837595363832898565092932586e-03
            6.85809565159406497869554186763708e-03 3.38109019285115053321533196140081e-01 1.78925917989028546856156154376549e-03
            9.93141904348405990532455689390190e-01 1.66223560954819097289225648950151e-03 1.11718522017194471957239262915884e-05
            9.64217441831786703509976632631151e-01 8.67282193331753159526442686910741e-03 1.33043156688835632721346069473611e-04
            9.13600657534882620858240898087388e-01 2.09410995388622407642476019873357e-02 4.86997688520965799768847048767384e-04
            8.43646452405842683930359271471389e-01 3.78962977032539991006920843119588e-02 1.14009972524951397679360276526950e-03
            7.57624318179077160806400570436381e-01 5.87459711381572505239923032149818e-02 2.08591522921379576502309838303972e-03
            6.59556184463944927820477914792718e-01 8.25153019122678438312945559118816e-02 3.24036075833260576659067275784309e-03
            5.54027474353671722795411369588692e-01 1.08092894976927825223533830012457e-01 4.45300338089021398302325138729429e-03
            4.45972525646328055159983705380000e-01 1.34282786843995111114580254252360e-01 5.53192422072981346281572712086927e-03
            3.40443815536054961157219622691628e-01 1.59860379908655064751243912724021e-01 6.27768777261338440515148207055063e-03
            2.42375681820922894704750660821446e-01 1.83629710682765651119652261513693e-01 6.52020900545657172697744385914120e-03
            1.56353547594157371580791959786438e-01 2.04479384117668888665164672602259e-01 6.15170620299740952136025029517441e-03
            8.63993424651176011863640269439202e-02 2.21434582282060588021010971715441e-01 5.14959252878980765916194783926585e-03
            3.57825581682132409788721361110220e-02 2.33702859887605390865061849581252e-01 3.58505760244088754126745577366364e-03
            6.85809565159406497869554186763708e-03 2.40713446211374693106677113974001e-01 1.61783024535910738631039329504802e-03
            9.93141904348405990532455689390190e-01 1.07228758486677903244421727890767e-03 9.46569859935521720035404152904235e-06
            9.64217441831786703509976632631151e-01 5.59472991159444196784811254019587e-03 1.12724944725771028660940564769533e-04
            9.13600657534882620858240898087388e-01 1.35088437042236320789756476301591e-02 4.12623910063243497237106538477747e-04
            8.43646452405842683930359271471389e-01 2.44464318452784247492903801912689e-02 9.65984885725451342813885347027281e-04
            7.57624318179077160806400570436381e-01 3.78962977032540129784798921264155e-02 1.76735643356469270329189846790996e-03
            6.59556184463944927820477914792718e-01 5.32295983155531213237843246588454e-02 2.74549624697271055923097904383212e-03
            5.54027474353671722795411369588692e-01 6.97293865143297497688479325006483e-02 3.77294535448017848838775378794708e-03
            4.45972525646328055159983705380000e-01 8.66241610798276495675196429147036e-02 4.68709453029124249734405083245292e-03
            3.40443815536054961157219622691628e-01 1.03123949278604271073689346849278e-01 5.31896585127309556861874284550140e-03
            2.42375681820922894704750660821446e-01 1.18457249890903365541205971567251e-01 5.52444949468222819277674062732331e-03
            1.56353547594157371580791959786438e-01 1.31907115748878950300948531548784e-01 5.21222405541627541170068838027873e-03
            8.63993424651176011863640269439202e-02 1.42844703889933699603176364689716e-01 4.36315213510554342213332290612016e-03
            3.57825581682132409788721361110220e-02 1.50758817682562940021284703107085e-01 3.03755134898844912985649280301459e-03
            6.85809565159406497869554186763708e-03 1.55281260009290572599027768774249e-01 1.37075689966013555300128334835108e-03
            9.93141904348405990532455689390190e-01 5.92534954860599906434348582706662e-04 7.31701647714658577023897831348442e-06
            9.64217441831786703509976632631151e-01 3.09158949745345175072475107924674e-03 8.71367569214689577762958383466696e-05
            9.13600657534882620858240898087388e-01 7.46484637840465464070360823711781e-03 3.18959653860420133241609930152549e-04
            8.43646452405842683930359271471389e-01 1.35088437042236615692747392358797e-02 7.46709527177334252828755367659141e-04
            7.57624318179077160806400570436381e-01 2.09410995388622962753988332451627e-02 1.36617239706583997599365076780487e-03
            6.59556184463944927820477914792718e-01 2.94141218086309448598303362132356e-02 2.12227772373946559775204079301147e-03
            5.54027474353671722795411369588692e-01 3.85317329733505520295011592679657e-02 2.91649929863450766212573661562146e-03
            4.45972525646328055159983705380000e-01 4.78676094917670630346506754904112e-02 3.62313964976877659132648012985101e-03
            3.40443815536054961157219622691628e-01 5.69852206564866667348745465915272e-02 4.11157828095184973982156861893600e-03
            2.42375681820922894704750660821446e-01 6.54582429262553083804121456523717e-02 4.27041782024493044717106116081595e-03
            1.56353547594157371580791959786438e-01 7.28904987608939430865362396616547e-02 4.02906651799145136549906709433344e-03
            8.63993424651176011863640269439202e-02 7.89344960867129213921700170430995e-02 3.37273110164729378671055748384333e-03
            3.57825581682132409788721361110220e-02 8.33077529676641559408523107777000e-02 2.34803728826114268754543523698430e-03
            6.85809565159406497869554186763708e-03 8.58068075102570021472914163496171e-02 1.05959963923410709768502258043554e-03
            9.93141904348405990532455689390190e-01 2.45400206576330966960070023574758e-04 4.82657129033289360425362096718160e-06
            9.64217441831786703509976632631151e-01 1.28039146906156614535987614544865e-03 5.74785871541303908889217200162136e-05
            9.13600657534882620858240898087388e-01 3.09158949745343917397955024739531e-03 2.10397436291898603339875495343847e-04
            8.43646452405842683930359271471389e-01 5.59472991159443155950725667935330e-03 4.92556874423991986923498931361109e-04
            7.57624318179077160806400570436381e-01 8.67282193331751771747661905465066e-03 9.01177206572956299331089713433585e-04
            6.59556184463944927820477914792718e-01 1.21819506324273487546383520907511e-02 1.39993189348516181708115269088921e-03
            5.54027474353671722795411369588692e-01 1.59580378403647096807738137158594e-02 1.92382944975338053550384387335725e-03
            4.45972525646328055159983705380000e-01 1.98245203278485347675452743487767e-02 2.38995523230805737013859868511645e-03
            3.40443815536054961157219622691628e-01 2.36006075357858939589572599970779e-02 2.71214719151997157853273279215500e-03
            2.42375681820922894704750660821446e-01 2.71097362348957232613955170563713e-02 2.81692355255676665781727763260278e-03
            1.56353547594157371580791959786438e-01 3.01878282566188094193648794316687e-02 2.65771941929022906606072673696417e-03
            8.63993424651176011863640269439202e-02 3.26909686707597957333604199448018e-02 2.22477660889071424815677247011081e-03
            3.57825581682132409788721361110220e-02 3.45021666991516817724061638728017e-02 1.54885114712381330957791103486443e-03
            6.85809565159406497869554186763708e-03 3.55371579616369087650795677291171e-02 6.98950619278751266415239395968229e-04
            9.93141904348405990532455689390190e-01 4.70334759664126607152899450792205e-05 2.11465349255878733207786659975547e-06
            9.64217441831786703509976632631151e-01 2.45400206576335303768759965592494e-04 2.51829482589996566852235532207871e-05
            9.13600657534882620858240898087388e-01 5.92534954860607929530424975439473e-04 9.21808975185347141126021575097127e-05
            8.43646452405842683930359271471389e-01 1.07228758486679594599810805277684e-03 2.15802699707496768374848095106699e-04
            7.57624318179077160806400570436381e-01 1.66223560954821786110613413001147e-03 3.94830494084018204879044100508168e-04
            6.59556184463944927820477914792718e-01 2.33479625093991130513337850516109e-03 6.13348625727779049425503199444165e-04
            5.54027474353671722795411369588692e-01 3.05852223886550608419887531397308e-03 8.42882539237879160019806246140206e-04
            4.45972525646328055159983705380000e-01 3.79957341272855976185840454206755e-03 1.04710505140198748171009857088620e-03
            3.40443815536054961157219622691628e-01 4.52329940065415454092390135087953e-03 1.18826620097134167015895389596380e-03
            2.42375681820922894704750660821446e-01 5.19586004204584776811071122892827e-03 1.23417160347680785782709111941813e-03
            1.56353547594157371580791959786438e-01 5.78580806672726859901656482065846e-03 1.16441989855199396625140551009281e-03
            8.63993424651176011863640269439202e-02 6.26556069673345553128207541249139e-03 9.74735758192719122655622854267676e-04
            3.57825581682132409788721361110220e-02 6.61269544501773016281775952052158e-03 6.78594242310084244142187603898719e-04
            6.85809565159406497869554186763708e-03 6.81106217562765185041806503818407e-03 3.06229470005817887116872366704001e-04
            ];
        
    case  26
        
        % ALG. DEG.:   26
        % PTS CARD.:  196
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.4e-15
        
        xyw=[
            9.93141904348405990532455689390190e-01 6.81106217562754169547734051093357e-03 2.11465349255880850790154795726394e-06
            9.64217441831786703509976632631151e-01 3.55371579616369642762307989869441e-02 2.51829482589999107951077295108888e-05
            9.13600657534882620858240898087388e-01 8.58068075102567801026864913183090e-02 9.21808975185356356844487701884816e-05
            8.43646452405842683930359271471389e-01 1.55281260009290517087876537516422e-01 2.15802699707498936779193066115567e-04
            7.57624318179077160806400570436381e-01 2.40713446211374693106677113974001e-01 3.94830494084022216427082296874573e-04
            6.59556184463944927820477914792718e-01 3.38109019285115164343835658655735e-01 6.13348625727785229377886366819439e-04
            5.54027474353671722795411369588692e-01 4.42914003407462730788068938636570e-01 8.42882539237887616796751633074791e-04
            4.45972525646328055159983705380000e-01 5.50227900940943426277840444527101e-01 1.04710505140199789005095443172877e-03
            3.40443815536054961157219622691628e-01 6.55032885063290937210922493250109e-01 1.18826620097135359638285123651258e-03
            2.42375681820922894704750660821446e-01 7.52428458137031408448081037931843e-01 1.23417160347682021773185745416868e-03
            1.56353547594157371580791959786438e-01 8.37860644339115445689003536244854e-01 1.16441989855200567563486835354070e-03
            8.63993424651176011863640269439202e-02 9.07335096838148946751800849597203e-01 9.74735758192728880475175223807582e-04
            3.57825581682132409788721361110220e-02 9.57604746386769178911890776362270e-01 6.78594242310091074615874262576654e-04
            6.85809565159406497869554186763708e-03 9.86330842172778443632807920948835e-01 3.06229470005820977093063950391638e-04
            9.93141904348405990532455689390190e-01 6.61269544501762260996224895848172e-03 4.82657129033283007678257689465617e-06
            9.64217441831786703509976632631151e-01 3.45021666991517303446634912234003e-02 5.74785871541296319474009801631098e-05
            9.13600657534882620858240898087388e-01 8.33077529676639338962473857463920e-02 2.10397436291895838624335657307540e-04
            8.43646452405842683930359271471389e-01 1.50758817682562884510133471849258e-01 4.92556874423985481710464018334505e-04
            7.57624318179077160806400570436381e-01 2.33702859887605363109486233952339e-01 9.01177206572944481527409621435254e-04
            6.59556184463944927820477914792718e-01 3.28261864903627686995690737603582e-01 1.39993189348514338564422043731383e-03
            5.54027474353671722795411369588692e-01 4.30014487805963463440406258087023e-01 1.92382944975335516517300771255350e-03
            4.45972525646328055159983705380000e-01 5.34202954025823406603024068317609e-01 2.38995523230802614511603110258875e-03
            3.40443815536054961157219622691628e-01 6.35955576928159072025437126285397e-01 2.71214719151993601670147526760957e-03
            2.42375681820922894704750660821446e-01 7.30514581944181395911641629936639e-01 2.81692355255672936126254413125025e-03
            1.56353547594157371580791959786438e-01 8.13458624149223874510994392039720e-01 2.65771941929019393791033820662051e-03
            8.63993424651176011863640269439202e-02 8.80909688864122575324699937482364e-01 2.22477660889068475785768086439020e-03
            3.57825581682132409788721361110220e-02 9.29715275132635077248721700016176e-01 1.54885114712379292657706830738107e-03
            6.85809565159406497869554186763708e-03 9.57604746386769067889588313846616e-01 6.98950619278742050696773269180540e-04
            9.93141904348405990532455689390190e-01 6.26556069673335578468220674608347e-03 7.31701647714661541639213221399629e-06
            9.64217441831786703509976632631151e-01 3.26909686707598512445116512026289e-02 8.71367569214693101420018961356107e-05
            9.13600657534882620858240898087388e-01 7.89344960867127409809285154551617e-02 3.18959653860421434284216912757870e-04
            8.43646452405842683930359271471389e-01 1.42844703889933699603176364689716e-01 7.46709527177337288594838327071557e-04
            7.57624318179077160806400570436381e-01 2.21434582282060643532162202973268e-01 1.36617239706584561384494769242792e-03
            6.59556184463944927820477914792718e-01 3.11029693727424227933653355648858e-01 2.12227772373947427136942067704695e-03
            5.54027474353671722795411369588692e-01 4.07440792672977780686238702401170e-01 2.91649929863451937150919945906935e-03
            4.45972525646328055159983705380000e-01 5.06159864861905006705455889459699e-01 3.62313964976879133647602593271131e-03
            3.40443815536054961157219622691628e-01 6.02570963807458559458041236212011e-01 4.11157828095186621969459039860340e-03
            2.42375681820922894704750660821446e-01 6.92166075252822032837229926371947e-01 4.27041782024494779440582092888690e-03
            1.56353547594157371580791959786438e-01 7.70755953644948976766215764655499e-01 4.02906651799146784537208887400084e-03
            8.63993424651176011863640269439202e-02 8.34666161448169630077131841972005e-01 3.37273110164730766449836529830009e-03
            3.57825581682132409788721361110220e-02 8.80909688864122908391607325029327e-01 2.34803728826115222852455310942332e-03
            6.85809565159406497869554186763708e-03 9.07335096838149168796405774628511e-01 1.05959963923411143449371252245328e-03
            9.93141904348405990532455689390190e-01 5.78580806672717492394886207307536e-03 9.46569859935521720035404152904235e-06
            9.64217441831786703509976632631151e-01 3.01878282566188545221752548286531e-02 1.12724944725771028660940564769533e-04
            9.13600657534882620858240898087388e-01 7.28904987608937487975069302592601e-02 4.12623910063243497237106538477747e-04
            8.43646452405842683930359271471389e-01 1.31907115748878894789797300290957e-01 9.65984885725451342813885347027281e-04
            7.57624318179077160806400570436381e-01 2.04479384117668888665164672602259e-01 1.76735643356469270329189846790996e-03
            6.59556184463944927820477914792718e-01 2.87214217220501943916843856641208e-01 2.74549624697271055923097904383212e-03
            5.54027474353671722795411369588692e-01 3.76243139131998471924589466652833e-01 3.77294535448017848838775378794708e-03
            4.45972525646328055159983705380000e-01 4.67403313273844323028072267334210e-01 4.68709453029124249734405083245292e-03
            3.40443815536054961157219622691628e-01 5.56432235185340795524666646088008e-01 5.31896585127309556861874284550140e-03
            2.42375681820922894704750660821446e-01 6.39167068288173823020770214498043e-01 5.52444949468222819277674062732331e-03
            1.56353547594157371580791959786438e-01 7.11739336656963761384986355551518e-01 5.21222405541627541170068838027873e-03
            8.63993424651176011863640269439202e-02 7.70755953644948754721610839624191e-01 4.36315213510554342213332290612016e-03
            3.57825581682132409788721361110220e-02 8.13458624149223874510994392039720e-01 3.03755134898844912985649280301459e-03
            6.85809565159406497869554186763708e-03 8.37860644339115445689003536244854e-01 1.37075689966013555300128334835108e-03
            9.93141904348405990532455689390190e-01 5.19586004204576363402212635378419e-03 1.11718522017194268669331921883803e-05
            9.64217441831786703509976632631151e-01 2.71097362348957683642058924533558e-02 1.33043156688835388775857260235114e-04
            9.13600657534882620858240898087388e-01 6.54582429262551418469584518788906e-02 4.86997688520964878197000436088615e-04
            8.43646452405842683930359271471389e-01 1.18457249890903323907842548123881e-01 1.14009972524951180838925779426063e-03
            7.57624318179077160806400570436381e-01 1.83629710682765651119652261513693e-01 2.08591522921379186189527743522376e-03
            6.59556184463944927820477914792718e-01 2.57928513623787214470439721480943e-01 3.24036075833260012873937583322004e-03
            5.54027474353671722795411369588692e-01 3.37879630669400410347691376955481e-01 4.45300338089020530940587150325882e-03
            4.45972525646328055159983705380000e-01 4.19744687509676861481011655996554e-01 5.53192422072980392183660924843025e-03
            3.40443815536054961157219622691628e-01 4.99695804555290001847112080213265e-01 6.27768777261337312944888822130451e-03
            2.42375681820922894704750660821446e-01 5.73994607496311592953475155809429e-01 6.52020900545655958391311202149154e-03
            1.56353547594157371580791959786438e-01 6.39167068288173823020770214498043e-01 6.15170620299739824565765644592830e-03
            8.63993424651176011863640269439202e-02 6.92166075252821810792625001340639e-01 5.14959252878979811818282996682683e-03
            3.57825581682132409788721361110220e-02 7.30514581944181506933944092452293e-01 3.58505760244088103605442086063704e-03
            6.85809565159406497869554186763708e-03 7.52428458137031408448081037931843e-01 1.61783024535910435054431033563560e-03
            9.93141904348405990532455689390190e-01 4.52329940065408081517617233657802e-03 1.23556468088325947275729621566143e-05
            9.64217441831786703509976632631151e-01 2.36006075357859286534267795332198e-02 1.47140709051487992209059085801925e-04
            9.13600657534882620858240898087388e-01 5.69852206564865071403147567252745e-02 5.38601059827556959656857937801533e-04
            8.43646452405842683930359271471389e-01 1.03123949278604215562538115591451e-01 1.26090725849935703269111275659498e-03
            7.57624318179077160806400570436381e-01 1.59860379908655036995668297095108e-01 2.30694350229267023180690365791179e-03
            6.59556184463944927820477914792718e-01 2.24541823999307582626627777244721e-01 3.58371667833169314440477926098083e-03
            5.54027474353671722795411369588692e-01 2.94143937391041054851825720106717e-01 4.92485364283183884442918198942607e-03
            4.45972525646328055159983705380000e-01 3.65412247072903983990954657201655e-01 6.11809936350968773532965627737212e-03
            3.40443815536054961157219622691628e-01 4.35014360464637428460576984434738e-01 6.94288570006323848410323407165379e-03
            2.42375681820922894704750660821446e-01 4.99695804555289946335960848955438e-01 7.21110502865334335870217685737771e-03
            1.56353547594157371580791959786438e-01 5.56432235185340684502364183572354e-01 6.80355483974647186340423132833166e-03
            8.63993424651176011863640269439202e-02 6.02570963807458226391133848665049e-01 5.69525494486377691777390097627176e-03
            3.57825581682132409788721361110220e-02 6.35955576928159072025437126285397e-01 3.96493837595365567622041069739680e-03
            6.85809565159406497869554186763708e-03 6.55032885063290826188620030734455e-01 1.78925917989029349165763793649830e-03
            9.93141904348405990532455689390190e-01 3.79957341272849687813240038281037e-03 1.29617156769339195052049770739977e-05
            9.64217441831786703509976632631151e-01 1.98245203278485590536739380240761e-02 1.54358251311008734529828245740646e-04
            9.13600657534882620858240898087388e-01 4.78676094917669173178786934386153e-02 5.65020505101317170940089518182958e-04
            8.43646452405842683930359271471389e-01 8.66241610798275801785806038424198e-02 1.32275724877207643236098899564013e-03
            7.57624318179077160806400570436381e-01 1.34282786843995055603429022994533e-01 2.42010363537519977358836165137745e-03
            6.59556184463944927820477914792718e-01 1.88615227280767905337910406160518e-01 3.75950505626425606625629072254924e-03
            5.54027474353671722795411369588692e-01 2.47081032014963314358269030890369e-01 5.16642743650341448868879012934485e-03
            4.45972525646328055159983705380000e-01 3.06946442338708491703869185585063e-01 6.41820421544856464096673320796071e-03
            3.40443815536054961157219622691628e-01 3.65412247072903872968652194686001e-01 7.28344795007722888013912410087869e-03
            2.42375681820922894704750660821446e-01 4.19744687509676694947557962223073e-01 7.56482396624481422814367803653113e-03
            1.56353547594157371580791959786438e-01 4.67403313273844156494618573560729e-01 7.13728263599933653027473923202706e-03
            8.63993424651176011863640269439202e-02 5.06159864861904673638548501912737e-01 5.97461844330196122782039580556557e-03
            3.57825581682132409788721361110220e-02 5.34202954025823184558419143286301e-01 4.15942643075042396694218638231177e-03
            6.85809565159406497869554186763708e-03 5.50227900940943204233235519495793e-01 1.87702587496293093159704934436149e-03
            9.93141904348405990532455689390190e-01 3.05852223886545534353720299236556e-03 1.29617156769339381399298166686052e-05
            9.64217441831786703509976632631151e-01 1.59580378403647304974555254375446e-02 1.54358251311008951370262742841533e-04
            9.13600657534882620858240898087388e-01 3.85317329733504410071986967523117e-02 5.65020505101317929881610258036062e-04
            8.43646452405842683930359271471389e-01 6.97293865143296942576967012428213e-02 1.32275724877207816708446497244722e-03
            7.57624318179077160806400570436381e-01 1.08092894976927783590170406569086e-01 2.42010363537520324303531360499164e-03
            6.59556184463944927820477914792718e-01 1.51828588255287111330460447788937e-01 3.75950505626426170410758764717229e-03
            5.54027474353671722795411369588692e-01 1.98891493631364796312865905747458e-01 5.16642743650342229494443202497678e-03
            4.45972525646328055159983705380000e-01 2.47081032014963314358269030890369e-01 6.41820421544857418194585108039973e-03
            3.40443815536054961157219622691628e-01 2.94143937391040999340674488848890e-01 7.28344795007723928847997996172126e-03
            2.42375681820922894704750660821446e-01 3.37879630669400299325388914439827e-01 7.56482396624482463648453389737369e-03
            1.56353547594157371580791959786438e-01 3.76243139131998360902287004137179e-01 7.13728263599934607125385710446608e-03
            8.63993424651176011863640269439202e-02 4.07440792672977503130482546112034e-01 5.97461844330196990143777568960104e-03
            3.57825581682132409788721361110220e-02 4.30014487805963352418103795571369e-01 4.15942643075043003847435230113661e-03
            6.85809565159406497869554186763708e-03 4.42914003407462564254615244863089e-01 1.87702587496293353368226330957214e-03
            9.93141904348405990532455689390190e-01 2.33479625093987270753603802120324e-03 1.23556468088325388233984433727919e-05
            9.64217441831786703509976632631151e-01 1.21819506324273643671496358820150e-02 1.47140709051487341687755594499265e-04
            9.13600657534882620858240898087388e-01 2.94141218086308615931034893264950e-02 5.38601059827554574412078469691778e-04
            8.43646452405842683930359271471389e-01 5.32295983155530866293148051227035e-02 1.26090725849935139483981583197192e-03
            7.57624318179077160806400570436381e-01 8.25153019122678299535067480974249e-02 2.30694350229266025714691679127100e-03
            6.59556184463944927820477914792718e-01 1.15901991536747461797318692333647e-01 3.58371667833167709821262647551521e-03
            5.54027474353671722795411369588692e-01 1.51828588255287111330460447788937e-01 4.92485364283181716038573227933739e-03
            4.45972525646328055159983705380000e-01 1.88615227280767905337910406160518e-01 6.11809936350966084711577863686216e-03
            3.40443815536054961157219622691628e-01 2.24541823999307554871052161615808e-01 6.94288570006320725908066648912609e-03
            2.42375681820922894704750660821446e-01 2.57928513623787158959288490223116e-01 7.21110502865331126631787128644646e-03
            1.56353547594157371580791959786438e-01 2.87214217220501888405692625383381e-01 6.80355483974644150574340173420751e-03
            8.63993424651176011863640269439202e-02 3.11029693727424005889048430617549e-01 5.69525494486375089692176132416535e-03
            3.57825581682132409788721361110220e-02 3.28261864903627631484539506345754e-01 3.96493837595363832898565092932586e-03
            6.85809565159406497869554186763708e-03 3.38109019285115053321533196140081e-01 1.78925917989028546856156154376549e-03
            9.93141904348405990532455689390190e-01 1.66223560954819097289225648950151e-03 1.11718522017194471957239262915884e-05
            9.64217441831786703509976632631151e-01 8.67282193331753159526442686910741e-03 1.33043156688835632721346069473611e-04
            9.13600657534882620858240898087388e-01 2.09410995388622407642476019873357e-02 4.86997688520965799768847048767384e-04
            8.43646452405842683930359271471389e-01 3.78962977032539991006920843119588e-02 1.14009972524951397679360276526950e-03
            7.57624318179077160806400570436381e-01 5.87459711381572505239923032149818e-02 2.08591522921379576502309838303972e-03
            6.59556184463944927820477914792718e-01 8.25153019122678438312945559118816e-02 3.24036075833260576659067275784309e-03
            5.54027474353671722795411369588692e-01 1.08092894976927825223533830012457e-01 4.45300338089021398302325138729429e-03
            4.45972525646328055159983705380000e-01 1.34282786843995111114580254252360e-01 5.53192422072981346281572712086927e-03
            3.40443815536054961157219622691628e-01 1.59860379908655064751243912724021e-01 6.27768777261338440515148207055063e-03
            2.42375681820922894704750660821446e-01 1.83629710682765651119652261513693e-01 6.52020900545657172697744385914120e-03
            1.56353547594157371580791959786438e-01 2.04479384117668888665164672602259e-01 6.15170620299740952136025029517441e-03
            8.63993424651176011863640269439202e-02 2.21434582282060588021010971715441e-01 5.14959252878980765916194783926585e-03
            3.57825581682132409788721361110220e-02 2.33702859887605390865061849581252e-01 3.58505760244088754126745577366364e-03
            6.85809565159406497869554186763708e-03 2.40713446211374693106677113974001e-01 1.61783024535910738631039329504802e-03
            9.93141904348405990532455689390190e-01 1.07228758486677903244421727890767e-03 9.46569859935521720035404152904235e-06
            9.64217441831786703509976632631151e-01 5.59472991159444196784811254019587e-03 1.12724944725771028660940564769533e-04
            9.13600657534882620858240898087388e-01 1.35088437042236320789756476301591e-02 4.12623910063243497237106538477747e-04
            8.43646452405842683930359271471389e-01 2.44464318452784247492903801912689e-02 9.65984885725451342813885347027281e-04
            7.57624318179077160806400570436381e-01 3.78962977032540129784798921264155e-02 1.76735643356469270329189846790996e-03
            6.59556184463944927820477914792718e-01 5.32295983155531213237843246588454e-02 2.74549624697271055923097904383212e-03
            5.54027474353671722795411369588692e-01 6.97293865143297497688479325006483e-02 3.77294535448017848838775378794708e-03
            4.45972525646328055159983705380000e-01 8.66241610798276495675196429147036e-02 4.68709453029124249734405083245292e-03
            3.40443815536054961157219622691628e-01 1.03123949278604271073689346849278e-01 5.31896585127309556861874284550140e-03
            2.42375681820922894704750660821446e-01 1.18457249890903365541205971567251e-01 5.52444949468222819277674062732331e-03
            1.56353547594157371580791959786438e-01 1.31907115748878950300948531548784e-01 5.21222405541627541170068838027873e-03
            8.63993424651176011863640269439202e-02 1.42844703889933699603176364689716e-01 4.36315213510554342213332290612016e-03
            3.57825581682132409788721361110220e-02 1.50758817682562940021284703107085e-01 3.03755134898844912985649280301459e-03
            6.85809565159406497869554186763708e-03 1.55281260009290572599027768774249e-01 1.37075689966013555300128334835108e-03
            9.93141904348405990532455689390190e-01 5.92534954860599906434348582706662e-04 7.31701647714658577023897831348442e-06
            9.64217441831786703509976632631151e-01 3.09158949745345175072475107924674e-03 8.71367569214689577762958383466696e-05
            9.13600657534882620858240898087388e-01 7.46484637840465464070360823711781e-03 3.18959653860420133241609930152549e-04
            8.43646452405842683930359271471389e-01 1.35088437042236615692747392358797e-02 7.46709527177334252828755367659141e-04
            7.57624318179077160806400570436381e-01 2.09410995388622962753988332451627e-02 1.36617239706583997599365076780487e-03
            6.59556184463944927820477914792718e-01 2.94141218086309448598303362132356e-02 2.12227772373946559775204079301147e-03
            5.54027474353671722795411369588692e-01 3.85317329733505520295011592679657e-02 2.91649929863450766212573661562146e-03
            4.45972525646328055159983705380000e-01 4.78676094917670630346506754904112e-02 3.62313964976877659132648012985101e-03
            3.40443815536054961157219622691628e-01 5.69852206564866667348745465915272e-02 4.11157828095184973982156861893600e-03
            2.42375681820922894704750660821446e-01 6.54582429262553083804121456523717e-02 4.27041782024493044717106116081595e-03
            1.56353547594157371580791959786438e-01 7.28904987608939430865362396616547e-02 4.02906651799145136549906709433344e-03
            8.63993424651176011863640269439202e-02 7.89344960867129213921700170430995e-02 3.37273110164729378671055748384333e-03
            3.57825581682132409788721361110220e-02 8.33077529676641559408523107777000e-02 2.34803728826114268754543523698430e-03
            6.85809565159406497869554186763708e-03 8.58068075102570021472914163496171e-02 1.05959963923410709768502258043554e-03
            9.93141904348405990532455689390190e-01 2.45400206576330966960070023574758e-04 4.82657129033289360425362096718160e-06
            9.64217441831786703509976632631151e-01 1.28039146906156614535987614544865e-03 5.74785871541303908889217200162136e-05
            9.13600657534882620858240898087388e-01 3.09158949745343917397955024739531e-03 2.10397436291898603339875495343847e-04
            8.43646452405842683930359271471389e-01 5.59472991159443155950725667935330e-03 4.92556874423991986923498931361109e-04
            7.57624318179077160806400570436381e-01 8.67282193331751771747661905465066e-03 9.01177206572956299331089713433585e-04
            6.59556184463944927820477914792718e-01 1.21819506324273487546383520907511e-02 1.39993189348516181708115269088921e-03
            5.54027474353671722795411369588692e-01 1.59580378403647096807738137158594e-02 1.92382944975338053550384387335725e-03
            4.45972525646328055159983705380000e-01 1.98245203278485347675452743487767e-02 2.38995523230805737013859868511645e-03
            3.40443815536054961157219622691628e-01 2.36006075357858939589572599970779e-02 2.71214719151997157853273279215500e-03
            2.42375681820922894704750660821446e-01 2.71097362348957232613955170563713e-02 2.81692355255676665781727763260278e-03
            1.56353547594157371580791959786438e-01 3.01878282566188094193648794316687e-02 2.65771941929022906606072673696417e-03
            8.63993424651176011863640269439202e-02 3.26909686707597957333604199448018e-02 2.22477660889071424815677247011081e-03
            3.57825581682132409788721361110220e-02 3.45021666991516817724061638728017e-02 1.54885114712381330957791103486443e-03
            6.85809565159406497869554186763708e-03 3.55371579616369087650795677291171e-02 6.98950619278751266415239395968229e-04
            9.93141904348405990532455689390190e-01 4.70334759664126607152899450792205e-05 2.11465349255878733207786659975547e-06
            9.64217441831786703509976632631151e-01 2.45400206576335303768759965592494e-04 2.51829482589996566852235532207871e-05
            9.13600657534882620858240898087388e-01 5.92534954860607929530424975439473e-04 9.21808975185347141126021575097127e-05
            8.43646452405842683930359271471389e-01 1.07228758486679594599810805277684e-03 2.15802699707496768374848095106699e-04
            7.57624318179077160806400570436381e-01 1.66223560954821786110613413001147e-03 3.94830494084018204879044100508168e-04
            6.59556184463944927820477914792718e-01 2.33479625093991130513337850516109e-03 6.13348625727779049425503199444165e-04
            5.54027474353671722795411369588692e-01 3.05852223886550608419887531397308e-03 8.42882539237879160019806246140206e-04
            4.45972525646328055159983705380000e-01 3.79957341272855976185840454206755e-03 1.04710505140198748171009857088620e-03
            3.40443815536054961157219622691628e-01 4.52329940065415454092390135087953e-03 1.18826620097134167015895389596380e-03
            2.42375681820922894704750660821446e-01 5.19586004204584776811071122892827e-03 1.23417160347680785782709111941813e-03
            1.56353547594157371580791959786438e-01 5.78580806672726859901656482065846e-03 1.16441989855199396625140551009281e-03
            8.63993424651176011863640269439202e-02 6.26556069673345553128207541249139e-03 9.74735758192719122655622854267676e-04
            3.57825581682132409788721361110220e-02 6.61269544501773016281775952052158e-03 6.78594242310084244142187603898719e-04
            6.85809565159406497869554186763708e-03 6.81106217562765185041806503818407e-03 3.06229470005817887116872366704001e-04
            ];
        
    case  27
        
        % ALG. DEG.:   27
        % PTS CARD.:  225
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.3e-15
        
        xyw=[
            9.93996259010242688702874147566035e-01 5.96769608388527435932635611948172e-03 1.41952736129656483203412883215000e-06
            9.68636696200352975694158885744400e-01 3.11750066470509283600431871263936e-02 1.69674225822881914488727411738367e-05
            9.24103291705213436557642125990242e-01 7.54410441162094991263131760206306e-02 6.25292760594908564233257441955516e-05
            8.62208865680085256855136321973987e-01 1.36963872038773165629166328471911e-01 1.47858527794593680795842693953546e-04
            7.85486086304269637281549876206554e-01 2.13226027719202099186546206510684e-01 2.74219417005117636378097589755498e-04
            6.97075673538781748206361044140067e-01 3.01105647265648423971384772812598e-01 4.33564553212285386848273693871647e-04
            6.00597046998717187804572859022301e-01 3.97005041120918888974955507364939e-01 6.09330292175421926910428016554988e-04
            5.00000000000000000000000000000000e-01 4.96998129505121288840285842525191e-01 7.78742212135606626244310746187693e-04
            3.99402953001282645661973447204218e-01 5.96991217889323966261372333974577e-01 9.16272579803009841990790729227001e-04
            3.02924326461218529349395112149068e-01 6.92890611744594098198035680979956e-01 9.97699017717176145358948247121589e-04
            2.14513913695730473740752586309100e-01 7.80770231291040506249601094168611e-01 1.00410986374296060975874755882842e-03
            1.37791134319915020700619834315148e-01 8.57032386971469328784678509691730e-01 9.25204180661707816304972773480131e-04
            7.58967082947862858866017177206231e-02 9.18555214894033356110014665318886e-01 7.61344083726070842803312910263003e-04
            3.13633037996470243058411142555997e-02 9.62821252363191826262323047558311e-01 5.24028599095731741422388161311119e-04
            6.00374098975725578597462117613759e-03 9.88028562926357434292867765179835e-01 2.35020945956645977911736444809776e-04
            9.93996259010242688702874147566035e-01 5.81544383716121274924271133954790e-03 3.24799999064399262226759956817546e-06
            9.68636696200352975694158885744400e-01 3.03796469744181257821580288691621e-02 3.88229138029352866369817154357236e-05
            9.24103291705213436557642125990242e-01 7.35163367751439783548406126101327e-02 1.43072330688081547417009331546467e-04
            8.62208865680085256855136321973987e-01 1.33469549113341329205084662135050e-01 3.38312955415547684211163836920377e-04
            7.85486086304269637281549876206554e-01 2.07786048651239951823299634270370e-01 6.27437475423869004710619723397258e-04
            6.97075673538781748206361044140067e-01 2.93423618782111594871508941650973e-01 9.92032773141360941712707521844550e-04
            6.00597046998717187804572859022301e-01 3.86876356847827451712618085366557e-01 1.39419981413898305504239782948162e-03
            5.00000000000000000000000000000000e-01 4.84318348100176432335928211614373e-01 1.78182877392393172570617121408532e-03
            3.99402953001282645661973447204218e-01 5.81760339352525801537296956666978e-01 2.09651001577684358706532563587643e-03
            3.02924326461218529349395112149068e-01 6.75213077418241103266893787804293e-01 2.28282066874081453547451481256303e-03
            2.14513913695730473740752586309100e-01 7.60850647549112912848556788958376e-01 2.29748923265827914227954487103034e-03
            1.37791134319915020700619834315148e-01 8.35167147087011452200044914206956e-01 2.11694628230924084683062247336238e-03
            7.58967082947862858866017177206231e-02 8.95120359425209177750559774722205e-01 1.74202036835732049745162353104888e-03
            3.13633037996470243058411142555997e-02 9.38257049225934891545364280318609e-01 1.19902224597172334975836438530905e-03
            6.00374098975725578597462117613759e-03 9.62821252363191826262323047558311e-01 5.37748021687375747125481328936303e-04
            9.93996259010242688702874147566035e-01 5.54807681118029902278543374904984e-03 4.94632226051372319577340702645429e-06
            9.68636696200352975694158885744400e-01 2.89829322800044947294750841138011e-02 5.91227350106579560627510783188399e-05
            9.24103291705213436557642125990242e-01 7.01363979648026436564123287098482e-02 2.17882344884404914684555842896430e-04
            8.62208865680085256855136321973987e-01 1.27333240792828483733956090873107e-01 5.15210870447175813338636096716527e-04
            7.85486086304269637281549876206554e-01 1.98233013762792537049151064820762e-01 9.55513534701172184866735381802982e-04
            6.97075673538781748206361044140067e-01 2.79933367220396467356380298951990e-01 1.51074932360925335943224734336354e-03
            6.00597046998717187804572859022301e-01 3.69089583585268110610400071891490e-01 2.12320246186713202860008031791494e-03
            5.00000000000000000000000000000000e-01 4.62051645852606662767669831737294e-01 2.71351581104417854256594111461709e-03
            3.99402953001282645661973447204218e-01 5.55013708119945547991846979130059e-01 3.19273835908197734650970645020607e-03
            3.02924326461218529349395112149068e-01 6.44169924484816802667808133264771e-01 3.47646758715497916289449698012959e-03
            2.14513913695730473740752586309100e-01 7.25870277942420871752915445540566e-01 3.49880608606006589592074540462363e-03
            1.37791134319915020700619834315148e-01 7.96770050912384841801383572601480e-01 3.22386039121318624528500507153694e-03
            7.58967082947862858866017177206231e-02 8.53966893740411014945834722311702e-01 2.65289228789858692533765172072435e-03
            3.13633037996470243058411142555997e-02 8.95120359425208955705954849690897e-01 1.82596996403475500536650599769928e-03
            6.00374098975725578597462117613759e-03 9.18555214894033134065409740287578e-01 8.18927037525052402625047598405672e-04
            9.93996259010242688702874147566035e-01 5.17647870861573117934328891465157e-03 6.44239056733915084533527092847116e-06
            9.68636696200352975694158885744400e-01 2.70417185930736173049560022718651e-02 7.70050413796523889383566219279942e-05
            9.24103291705213436557642125990242e-01 6.54388147677002346425112477845687e-02 2.83783200839655795622018530366404e-04
            8.62208865680085256855136321973987e-01 1.18804737622746001135531912495935e-01 6.71041933206905348054116533518254e-04
            7.85486086304269637281549876206554e-01 1.84955798200191423585891925540636e-01 1.24451886850663554634488505712397e-03
            6.97075673538781748206361044140067e-01 2.61184039905030807027230821404373e-01 1.96769168675706325466467916385227e-03
            6.00597046998717187804572859022301e-01 3.44368767056512481428143246375839e-01 2.76538785636329170411484845715222e-03
            5.00000000000000000000000000000000e-01 4.31104432840042572916416929729166e-01 3.53424781982988661307998334848435e-03
            3.99402953001282645661973447204218e-01 5.17840098623572941960446769371629e-01 4.15841637588638176475930308129136e-03
            3.02924326461218529349395112149068e-01 6.01024825775054227783300575538306e-01 4.52796255713864854303141171953939e-03
            2.14513913695730473740752586309100e-01 6.77253067479893777758093165175524e-01 4.55705757502365346733874673645914e-03
            1.37791134319915020700619834315148e-01 7.43404128057339130819514139147941e-01 4.19895160098465551612445523232964e-03
            7.58967082947862858866017177206231e-02 7.96770050912385174868290960148443e-01 3.45528806082068219371339345968863e-03
            3.13633037996470243058411142555997e-02 8.35167147087011674244649839238264e-01 2.37825419634513481395576839361183e-03
            6.00374098975725578597462117613759e-03 8.57032386971469550829283434723038e-01 1.06662031788896114262166214814442e-03
            9.93996259010242688702874147566035e-01 4.71585501322903648951934130195696e-03 7.67475790123426755359476586493273e-06
            9.68636696200352975694158885744400e-01 2.46354387551566134106906247325242e-02 9.17353649372818115296027152716363e-05
            9.24103291705213436557642125990242e-01 5.96158083618486961396953915937047e-02 3.38068196908658990997714743897973e-04
            8.62208865680085256855136321973987e-01 1.08233018824375804012305479773204e-01 7.99405798997733312537261607388928e-04
            7.85486086304269637281549876206554e-01 1.68497694526671154546804132223770e-01 1.48258335465236519221110533806041e-03
            6.97075673538781748206361044140067e-01 2.37942843638379225579271292190242e-01 2.34409217545607923005590578213742e-03
            6.00597046998717187804572859022301e-01 3.13725462411345801250206477561733e-01 3.29437994774776834322471685823075e-03
            5.00000000000000000000000000000000e-01 3.92743043152134763129623706845450e-01 4.21031542509575380101027874957254e-03
            3.99402953001282645661973447204218e-01 4.71760623892924058075948323676130e-01 4.95388142085857914598268081363130e-03
            3.02924326461218529349395112149068e-01 5.47543242665890272924400505871745e-01 5.39411823121517097073818192143335e-03
            2.14513913695730473740752586309100e-01 6.16988391777598454979170128353871e-01 5.42877884610113315561052971247591e-03
            1.37791134319915020700619834315148e-01 6.77253067479893666735790702659870e-01 5.00217064453254388334269009419586e-03
            7.58967082947862858866017177206231e-02 7.25870277942421093797520370571874e-01 4.11625142385254206428557566255222e-03
            3.13633037996470243058411142555997e-02 7.60850647549113023870859251474030e-01 2.83319134314485854275744713959284e-03
            6.00374098975725578597462117613759e-03 7.80770231291040617271903556684265e-01 1.27065452284684000938685954906759e-03
            9.93996259010242688702874147566035e-01 4.18506179418750862836207105033282e-03 8.59293576793571239349177337540908e-06
            9.68636696200352975694158885744400e-01 2.18625961205404194254153793508522e-02 1.02710223397070615119000547288408e-04
            9.24103291705213436557642125990242e-01 5.29057490539647859439931210090435e-02 3.78513347079099699444421078808887e-04
            8.62208865680085256855136321973987e-01 9.60508477637273538896067748282803e-02 8.95043566416359008541814290538241e-04
            7.85486086304269637281549876206554e-01 1.49532430872891392015233691381582e-01 1.65995379933608043498960427086786e-03
            6.97075673538781748206361044140067e-01 2.11161178899235524308863887199550e-01 2.62453014896735789887594059166531e-03
            6.00597046998717187804572859022301e-01 2.78414082476747593020149906806182e-01 3.68850661486264932639733693520157e-03
            5.00000000000000000000000000000000e-01 3.48537836769390818592029290812206e-01 4.71402101228214458206178250065932e-03
            3.99402953001282645661973447204218e-01 4.18661591062034321719664831107366e-01 5.54654432090450049547269983918341e-03
            3.02924326461218529349395112149068e-01 4.85914494639546057364043463167036e-01 6.03944933273122073341676596669458e-03
            2.14513913695730473740752586309100e-01 5.47543242665890383946702968387399e-01 6.07825660733519578665662308480933e-03
            1.37791134319915020700619834315148e-01 6.01024825775054227783300575538306e-01 5.60061067747936146277165647688889e-03
            7.58967082947862858866017177206231e-02 6.44169924484817135734715520811733e-01 4.60870356368510772360203020525660e-03
            3.13633037996470243058411142555997e-02 6.75213077418241325311498712835601e-01 3.17214321848510727441805556736654e-03
            6.00374098975725578597462117613759e-03 6.92890611744594320242640606011264e-01 1.42267063516152625010646737280240e-03
            9.93996259010242688702874147566035e-01 3.60582910939343014369851125877631e-03 9.15932449658844526985471135027694e-06
            9.68636696200352975694158885744400e-01 1.88367076461916811702668184125287e-02 1.09480192872064160621893569302898e-04
            9.24103291705213436557642125990242e-01 4.55833388787718549184369010163209e-02 4.03462409799921583979098738126368e-04
            8.62208865680085256855136321973987e-01 8.27569483751444262553675912386097e-02 9.54038838970706535268229142587870e-04
            7.85486086304269637281549876206554e-01 1.28836423105793351773229460377479e-01 1.76936682736504417856604476355642e-03
            6.97075673538781748206361044140067e-01 1.81935455936683060373937337317329e-01 2.79752158455229922237594841760711e-03
            6.00597046998717187804572859022301e-01 2.39880234135137898610068418747687e-01 3.93162824740350604763028385946200e-03
            5.00000000000000000000000000000000e-01 3.00298523499358538391135198253323e-01 5.02473768003051327740715592540255e-03
            3.99402953001282645661973447204218e-01 3.60716812863579427972382518419181e-01 5.91213534487741894374313034177248e-03
            3.02924326461218529349395112149068e-01 4.18661591062033988652757443560404e-01 6.43752935121481937447152077425017e-03
            2.14513913695730473740752586309100e-01 4.71760623892923780520192167386995e-01 6.47889454124130331064135290830563e-03
            1.37791134319915020700619834315148e-01 5.17840098623572719915841844340321e-01 5.96976539327895294739789733284852e-03
            7.58967082947862858866017177206231e-02 5.55013708119945436969544516614405e-01 4.91247841114914388094270236706507e-03
            3.13633037996470243058411142555997e-02 5.81760339352525579492692031635670e-01 3.38122963704375360458054267098760e-03
            6.00374098975725578597462117613759e-03 5.96991217889323744216767408943269e-01 1.51644354748183864123323161265944e-03
            9.93996259010242688702874147566035e-01 3.00187049487868340413854184589582e-03 9.35073307890573744195501088016798e-06
            9.68636696200352975694158885744400e-01 1.56816518998235399084961727567133e-02 1.11768074311057922067338954441595e-04
            9.24103291705213436557642125990242e-01 3.79483541473932817211789370048791e-02 4.11893835928222402844450789061170e-04
            8.62208865680085256855136321973987e-01 6.88955671599573993280074546419200e-02 9.73976032124092001678039576262336e-04
            7.85486086304269637281549876206554e-01 1.07256956847865209114800677525636e-01 1.80634247946158616252021467118993e-03
            6.97075673538781748206361044140067e-01 1.51462163230609125896819477929967e-01 2.85598328013919816950605401473240e-03
            6.00597046998717187804572859022301e-01 1.99701476500641406097713570488850e-01 4.01379013492200471119319971080586e-03
            5.00000000000000000000000000000000e-01 2.49999999999999972244424384371086e-01 5.12974301270642790012699308022093e-03
            3.99402953001282645661973447204218e-01 3.00298523499358704924588892026804e-01 6.03568522513905220128194173412339e-03
            3.02924326461218529349395112149068e-01 3.48537836769390763080878059554379e-01 6.57205874442508795885142802717382e-03
            2.14513913695730473740752586309100e-01 3.92743043152134763129623706845450e-01 6.61428837072994615930987549745623e-03
            1.37791134319915020700619834315148e-01 4.31104432840042517405265698471339e-01 6.09451960753568805495561377938429e-03
            7.58967082947862858866017177206231e-02 4.62051645852606829301123525510775e-01 5.01513778616010513777823121017718e-03
            3.13633037996470243058411142555997e-02 4.84318348100176487847079442872200e-01 3.45188947353680707433309926557286e-03
            6.00374098975725578597462117613759e-03 4.96998129505121344351437073783018e-01 1.54813369119230753029325953207263e-03
            9.93996259010242688702874147566035e-01 2.39791188036393579721683444461178e-03 9.15932449658848592743617955669322e-06
            9.68636696200352975694158885744400e-01 1.25265961534553917078316231936697e-02 1.09480192872064634960344031711088e-04
            9.24103291705213436557642125990242e-01 3.03133694160146981155801171325948e-02 4.03462409799923372912683339208684e-04
            8.62208865680085256855136321973987e-01 5.50341859447703515839656063235452e-02 9.54038838970710763656701836055163e-04
            7.85486086304269637281549876206554e-01 8.56774905899370109452206634159666e-02 1.76936682736505198482168665918834e-03
            6.97075673538781748206361044140067e-01 1.20988870524535149786338195099233e-01 2.79752158455231136544028025525677e-03
            6.00597046998717187804572859022301e-01 1.59522718866144858074207490972185e-01 3.93162824740352339486504362753294e-03
            5.00000000000000000000000000000000e-01 1.99701476500641295075411107973196e-01 5.02473768003053496145060563549123e-03
            3.99402953001282645661973447204218e-01 2.39880234135137870854492803118774e-01 5.91213534487744409723353200547535e-03
            3.02924326461218529349395112149068e-01 2.78414082476747426486696213032701e-01 6.43752935121484799740887439156722e-03
            2.14513913695730473740752586309100e-01 3.13725462411345579205601552530425e-01 6.47889454124133193357870652562269e-03
            1.37791134319915020700619834315148e-01 3.44368767056512203872387090086704e-01 5.96976539327897896825003698495493e-03
            7.58967082947862858866017177206231e-02 3.69089583585268055099248840633663e-01 4.91247841114916469762441408875020e-03
            3.13633037996470243058411142555997e-02 3.86876356847827285179164391593076e-01 3.38122963704376834973008847384790e-03
            6.00374098975725578597462117613759e-03 3.97005041120918722441501813591458e-01 1.51644354748184536328670102278693e-03
            9.93996259010242688702874147566035e-01 1.81867919556985991463848861826591e-03 8.59293576793570900535998435820773e-06
            9.68636696200352975694158885744400e-01 9.50070767910666733047087006980291e-03 1.02710223397070560908891923013186e-04
            9.24103291705213436557642125990242e-01 2.29909592408217983150464647223998e-02 3.78513347079099536814095205983222e-04
            8.62208865680085256855136321973987e-01 4.17402865561874863997715578989300e-02 8.95043566416358574860945296336467e-04
            7.85486086304269637281549876206554e-01 6.49814828228390956033067027419747e-02 1.65995379933607956762786628246431e-03
            6.97075673538781748206361044140067e-01 9.17631475619828107515019155471236e-02 2.62453014896735659783333360905999e-03
            6.00597046998717187804572859022301e-01 1.20988870524535316319791888872714e-01 3.68850661486264802535472995259624e-03
            5.00000000000000000000000000000000e-01 1.51462163230609236919121940445621e-01 4.71402101228214284733830652385222e-03
            3.99402953001282645661973447204218e-01 1.81935455936683282418542262348637e-01 5.54654432090449789338748587397276e-03
            3.02924326461218529349395112149068e-01 2.11161178899235635331166349715204e-01 6.03944933273121726396981401308039e-03
            2.14513913695730473740752586309100e-01 2.37942843638379419868300601592637e-01 6.07825660733519318457140911959868e-03
            1.37791134319915020700619834315148e-01 2.61184039905030973560684515177854e-01 5.60061067747935886068644251167825e-03
            7.58967082947862858866017177206231e-02 2.79933367220396800423287686498952e-01 4.60870356368510598887855422844950e-03
            3.13633037996470243058411142555997e-02 2.93423618782111872427265097940108e-01 3.17214321848510597337544858476122e-03
            6.00374098975725578597462117613759e-03 3.01105647265648701527140929101733e-01 1.42267063516152559958516388149974e-03
            9.93996259010242688702874147566035e-01 1.28788597652833118611948037823822e-03 7.67475790123429804678086701974493e-06
            9.68636696200352975694158885744400e-01 6.72786504449046900838693474611318e-03 9.17353649372821638953087730605773e-05
            9.24103291705213436557642125990242e-01 1.62808999329378777110033382768961e-02 3.38068196908660292040321726503294e-04
            8.62208865680085256855136321973987e-01 2.95581154955390085214972373250930e-02 7.99405798997736456723561815351786e-04
            7.85486086304269637281549876206554e-01 4.60162191690592914383728384564165e-02 1.48258335465237083006240226268346e-03
            6.97075673538781748206361044140067e-01 6.49814828228390539699432792986045e-02 2.34409217545608833735415466037466e-03
            6.00597046998717187804572859022301e-01 8.56774905899370803341597024882503e-02 3.29437994774778091996991769008218e-03
            5.00000000000000000000000000000000e-01 1.07256956847865222992588485340093e-01 4.21031542509576941352156254083638e-03
            3.99402953001282645661973447204218e-01 1.28836423105793435039956307264220e-01 4.95388142085859822794091655850934e-03
            3.02924326461218529349395112149068e-01 1.49532430872891364259658075752668e-01 5.39411823121519178741989364311848e-03
            2.14513913695730473740752586309100e-01 1.68497694526671182302379747852683e-01 5.42877884610115397229224143416104e-03
            1.37791134319915020700619834315148e-01 1.84955798200191423585891925540636e-01 5.00217064453256296530092583907390e-03
            7.58967082947862858866017177206231e-02 1.98233013762792648071453527336416e-01 4.11625142385255767679685945381607e-03
            3.13633037996470243058411142555997e-02 2.07786048651240007334450865528197e-01 2.83319134314486938477917199463718e-03
            6.00374098975725578597462117613759e-03 2.13226027719202154697697437768511e-01 1.27065452284684499671685298238799e-03
            9.93996259010242688702874147566035e-01 8.27262281141636929976401759745386e-04 6.44239056733915846863179621717421e-06
            9.68636696200352975694158885744400e-01 4.32158520657347205301546111400057e-03 7.70050413796524838060467144096322e-05
            9.24103291705213436557642125990242e-01 1.04578935270863444123579100164534e-02 2.83783200839656120882670276017734e-04
            8.62208865680085256855136321973987e-01 1.89863966971688322149525163240469e-02 6.71041933206906106995637273371358e-04
            7.85486086304269637281549876206554e-01 2.95581154955390501548606607684633e-02 1.24451886850663706422792653683018e-03
            6.97075673538781748206361044140067e-01 4.17402865561875141553471735278436e-02 1.96769168675706542306902413486114e-03
            6.00597046998717187804572859022301e-01 5.50341859447704695451619727464276e-02 2.76538785636329473988093141656464e-03
            5.00000000000000000000000000000000e-01 6.88955671599574964725221093431173e-02 3.53424781982989094988867329050208e-03
            3.99402953001282645661973447204218e-01 8.27569483751445789110334771976341e-02 4.15841637588638696892973101171265e-03
            3.02924326461218529349395112149068e-01 9.60508477637274787896970451583911e-02 4.52796255713865374720183964996068e-03
            2.14513913695730473740752586309100e-01 1.08233018824375970545759173546685e-01 4.55705757502365953887091265528397e-03
            1.37791134319915020700619834315148e-01 1.18804737622746167668985606269416e-01 4.19895160098465985293314517434737e-03
            7.58967082947862858866017177206231e-02 1.27333240792828705778561015904415e-01 3.45528806082068653052208340170637e-03
            3.13633037996470243058411142555997e-02 1.33469549113341551249689587166358e-01 2.37825419634513741604098235882248e-03
            6.00374098975725578597462117613759e-03 1.36963872038773387673771253503219e-01 1.06662031788896244366426913074974e-03
            9.93996259010242688702874147566035e-01 4.55664178577066376028825711586023e-04 4.94632226051369185555435861734175e-06
            9.68636696200352975694158885744400e-01 2.38037151964257684758075050979187e-03 5.91227350106575901445178644610934e-05
            9.24103291705213436557642125990242e-01 5.76031032998390243871078553183906e-03 2.17882344884403559431840236015887e-04
            8.62208865680085256855136321973987e-01 1.04578935270862889012066787586264e-02 5.15210870447172560732118640203225e-04
            7.85486086304269637281549876206554e-01 1.62808999329378117915112511582265e-02 9.55513534701166221754786711528595e-04
            6.97075673538781748206361044140067e-01 2.29909592408216907621909541603600e-02 1.51074932360924403529356396802541e-03
            6.00597046998717187804572859022301e-01 3.03133694160145975016185104777833e-02 2.12320246186711901817401049186174e-03
            5.00000000000000000000000000000000e-01 3.79483541473931360044069549530832e-02 2.71351581104416162901205034074792e-03
            3.99402953001282645661973447204218e-01 4.55833388787717022627710150572966e-02 3.19273835908195783087060171112626e-03
            3.02924326461218529349395112149068e-01 5.29057490539645777771760037921922e-02 3.47646758715495747885104727004091e-03
            2.14513913695730473740752586309100e-01 5.96158083618484671561965626551682e-02 3.49880608606004421187729569453495e-03
            1.37791134319915020700619834315148e-01 6.54388147676999848423307071243471e-02 3.22386039121316586228416234405358e-03
            7.58967082947862858866017177206231e-02 7.01363979648023938562317880496266e-02 2.65289228789857044546462994105696e-03
            3.13633037996470243058411142555997e-02 7.35163367751437146768722641354543e-02 1.82596996403474351282347765135228e-03
            6.00374098975725578597462117613759e-03 7.54410441162092215705570197314955e-02 8.18927037525047306874836916534832e-04
            9.93996259010242688702874147566035e-01 1.88297152596153815088883543005238e-04 3.24799999064399304578407319532563e-06
            9.68636696200352975694158885744400e-01 9.83656825228955119036489129769052e-04 3.88229138029352934132452934701263e-05
            9.24103291705213436557642125990242e-01 2.38037151964258118438944045180961e-03 1.43072330688081574522063643684078e-04
            8.62208865680085256855136321973987e-01 4.32158520657345730786591531114027e-03 3.38312955415547738421272461195599e-04
            7.85486086304269637281549876206554e-01 6.72786504449045599796086492005998e-03 6.27437475423869113130836971947701e-04
            6.97075673538781748206361044140067e-01 9.50070767910664130961873041769650e-03 9.92032773141360941712707521844550e-04
            6.00597046998717187804572859022301e-01 1.25265961534553743605968634255987e-02 1.39419981413898327188283232658250e-03
            5.00000000000000000000000000000000e-01 1.56816518998235086834736051741856e-02 1.78182877392393215938704020828709e-03
            3.99402953001282645661973447204218e-01 1.88367076461916568841381547372293e-02 2.09651001577684402074619463007821e-03
            3.02924326461218529349395112149068e-01 2.18625961205403743226050039538677e-02 2.28282066874081540283625280096658e-03
            2.14513913695730473740752586309100e-01 2.46354387551565683078802493355397e-02 2.29748923265827957596041386523211e-03
            1.37791134319915020700619834315148e-01 2.70417185930735617938047710140381e-02 2.11694628230924128051149146756416e-03
            7.58967082947862858866017177206231e-02 2.89829322800044496266647087168167e-02 1.74202036835732093113249252525065e-03
            3.13633037996470243058411142555997e-02 3.03796469744180702710067976113351e-02 1.19902224597172356659879888240994e-03
            6.00374098975725578597462117613759e-03 3.11750066470508728488919558685666e-02 5.37748021687375855545698577486746e-04
            9.93996259010242688702874147566035e-01 3.60449058720920965850215145209745e-05 1.41952736129656462027589201857491e-06
            9.68636696200352975694158885744400e-01 1.88297152596150670902583335042380e-04 1.69674225822881880607409521566353e-05
            9.24103291705213436557642125990242e-01 4.55664178577059599765247677183311e-04 6.25292760594908428707985881267462e-05
            8.62208865680085256855136321973987e-01 8.27262281141620233262945482977102e-04 1.47858527794593653690788381815935e-04
            7.85486086304269637281549876206554e-01 1.28788597652830711683125120003979e-03 2.74219417005117582167988965480276e-04
            6.97075673538781748206361044140067e-01 1.81867919556982456964766559082136e-03 4.33564553212285278428056445321204e-04
            6.00597046998717187804572859022301e-01 2.39791188036389242912993502443442e-03 6.09330292175421818490210768004545e-04
            5.00000000000000000000000000000000e-01 3.00187049487862745930644159386702e-03 7.78742212135606409403876249086807e-04
            3.99402953001282645661973447204218e-01 3.60582910939336465788729313430849e-03 9.16272579803009625150356232126114e-04
            3.02924326461218529349395112149068e-01 4.18506179418742969844391410561002e-03 9.97699017717175928518513750020702e-04
            2.14513913695730473740752586309100e-01 4.71585501322894801862206648479514e-03 1.00410986374296039291831306172753e-03
            1.37791134319915020700619834315148e-01 5.17647870861563490219037220185783e-03 9.25204180661707599464538276379244e-04
            7.58967082947862858866017177206231e-02 5.54807681118019754146208910583482e-03 7.61344083726070734383095661712559e-04
            3.13633037996470243058411142555997e-02 5.81544383716110519638720077750804e-03 5.24028599095731633002170912760675e-04
            6.00374098975725578597462117613759e-03 5.96769608388516420438563159223122e-03 2.35020945956645923701627820534554e-04
            ];
        
    case  28
        
        % ALG. DEG.:   28
        % PTS CARD.:  225
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.3e-15
        
        xyw=[
            9.93996259010242688702874147566035e-01 5.96769608388527435932635611948172e-03 1.41952736129656483203412883215000e-06
            9.68636696200352975694158885744400e-01 3.11750066470509283600431871263936e-02 1.69674225822881914488727411738367e-05
            9.24103291705213436557642125990242e-01 7.54410441162094991263131760206306e-02 6.25292760594908564233257441955516e-05
            8.62208865680085256855136321973987e-01 1.36963872038773165629166328471911e-01 1.47858527794593680795842693953546e-04
            7.85486086304269637281549876206554e-01 2.13226027719202099186546206510684e-01 2.74219417005117636378097589755498e-04
            6.97075673538781748206361044140067e-01 3.01105647265648423971384772812598e-01 4.33564553212285386848273693871647e-04
            6.00597046998717187804572859022301e-01 3.97005041120918888974955507364939e-01 6.09330292175421926910428016554988e-04
            5.00000000000000000000000000000000e-01 4.96998129505121288840285842525191e-01 7.78742212135606626244310746187693e-04
            3.99402953001282645661973447204218e-01 5.96991217889323966261372333974577e-01 9.16272579803009841990790729227001e-04
            3.02924326461218529349395112149068e-01 6.92890611744594098198035680979956e-01 9.97699017717176145358948247121589e-04
            2.14513913695730473740752586309100e-01 7.80770231291040506249601094168611e-01 1.00410986374296060975874755882842e-03
            1.37791134319915020700619834315148e-01 8.57032386971469328784678509691730e-01 9.25204180661707816304972773480131e-04
            7.58967082947862858866017177206231e-02 9.18555214894033356110014665318886e-01 7.61344083726070842803312910263003e-04
            3.13633037996470243058411142555997e-02 9.62821252363191826262323047558311e-01 5.24028599095731741422388161311119e-04
            6.00374098975725578597462117613759e-03 9.88028562926357434292867765179835e-01 2.35020945956645977911736444809776e-04
            9.93996259010242688702874147566035e-01 5.81544383716121274924271133954790e-03 3.24799999064399262226759956817546e-06
            9.68636696200352975694158885744400e-01 3.03796469744181257821580288691621e-02 3.88229138029352866369817154357236e-05
            9.24103291705213436557642125990242e-01 7.35163367751439783548406126101327e-02 1.43072330688081547417009331546467e-04
            8.62208865680085256855136321973987e-01 1.33469549113341329205084662135050e-01 3.38312955415547684211163836920377e-04
            7.85486086304269637281549876206554e-01 2.07786048651239951823299634270370e-01 6.27437475423869004710619723397258e-04
            6.97075673538781748206361044140067e-01 2.93423618782111594871508941650973e-01 9.92032773141360941712707521844550e-04
            6.00597046998717187804572859022301e-01 3.86876356847827451712618085366557e-01 1.39419981413898305504239782948162e-03
            5.00000000000000000000000000000000e-01 4.84318348100176432335928211614373e-01 1.78182877392393172570617121408532e-03
            3.99402953001282645661973447204218e-01 5.81760339352525801537296956666978e-01 2.09651001577684358706532563587643e-03
            3.02924326461218529349395112149068e-01 6.75213077418241103266893787804293e-01 2.28282066874081453547451481256303e-03
            2.14513913695730473740752586309100e-01 7.60850647549112912848556788958376e-01 2.29748923265827914227954487103034e-03
            1.37791134319915020700619834315148e-01 8.35167147087011452200044914206956e-01 2.11694628230924084683062247336238e-03
            7.58967082947862858866017177206231e-02 8.95120359425209177750559774722205e-01 1.74202036835732049745162353104888e-03
            3.13633037996470243058411142555997e-02 9.38257049225934891545364280318609e-01 1.19902224597172334975836438530905e-03
            6.00374098975725578597462117613759e-03 9.62821252363191826262323047558311e-01 5.37748021687375747125481328936303e-04
            9.93996259010242688702874147566035e-01 5.54807681118029902278543374904984e-03 4.94632226051372319577340702645429e-06
            9.68636696200352975694158885744400e-01 2.89829322800044947294750841138011e-02 5.91227350106579560627510783188399e-05
            9.24103291705213436557642125990242e-01 7.01363979648026436564123287098482e-02 2.17882344884404914684555842896430e-04
            8.62208865680085256855136321973987e-01 1.27333240792828483733956090873107e-01 5.15210870447175813338636096716527e-04
            7.85486086304269637281549876206554e-01 1.98233013762792537049151064820762e-01 9.55513534701172184866735381802982e-04
            6.97075673538781748206361044140067e-01 2.79933367220396467356380298951990e-01 1.51074932360925335943224734336354e-03
            6.00597046998717187804572859022301e-01 3.69089583585268110610400071891490e-01 2.12320246186713202860008031791494e-03
            5.00000000000000000000000000000000e-01 4.62051645852606662767669831737294e-01 2.71351581104417854256594111461709e-03
            3.99402953001282645661973447204218e-01 5.55013708119945547991846979130059e-01 3.19273835908197734650970645020607e-03
            3.02924326461218529349395112149068e-01 6.44169924484816802667808133264771e-01 3.47646758715497916289449698012959e-03
            2.14513913695730473740752586309100e-01 7.25870277942420871752915445540566e-01 3.49880608606006589592074540462363e-03
            1.37791134319915020700619834315148e-01 7.96770050912384841801383572601480e-01 3.22386039121318624528500507153694e-03
            7.58967082947862858866017177206231e-02 8.53966893740411014945834722311702e-01 2.65289228789858692533765172072435e-03
            3.13633037996470243058411142555997e-02 8.95120359425208955705954849690897e-01 1.82596996403475500536650599769928e-03
            6.00374098975725578597462117613759e-03 9.18555214894033134065409740287578e-01 8.18927037525052402625047598405672e-04
            9.93996259010242688702874147566035e-01 5.17647870861573117934328891465157e-03 6.44239056733915084533527092847116e-06
            9.68636696200352975694158885744400e-01 2.70417185930736173049560022718651e-02 7.70050413796523889383566219279942e-05
            9.24103291705213436557642125990242e-01 6.54388147677002346425112477845687e-02 2.83783200839655795622018530366404e-04
            8.62208865680085256855136321973987e-01 1.18804737622746001135531912495935e-01 6.71041933206905348054116533518254e-04
            7.85486086304269637281549876206554e-01 1.84955798200191423585891925540636e-01 1.24451886850663554634488505712397e-03
            6.97075673538781748206361044140067e-01 2.61184039905030807027230821404373e-01 1.96769168675706325466467916385227e-03
            6.00597046998717187804572859022301e-01 3.44368767056512481428143246375839e-01 2.76538785636329170411484845715222e-03
            5.00000000000000000000000000000000e-01 4.31104432840042572916416929729166e-01 3.53424781982988661307998334848435e-03
            3.99402953001282645661973447204218e-01 5.17840098623572941960446769371629e-01 4.15841637588638176475930308129136e-03
            3.02924326461218529349395112149068e-01 6.01024825775054227783300575538306e-01 4.52796255713864854303141171953939e-03
            2.14513913695730473740752586309100e-01 6.77253067479893777758093165175524e-01 4.55705757502365346733874673645914e-03
            1.37791134319915020700619834315148e-01 7.43404128057339130819514139147941e-01 4.19895160098465551612445523232964e-03
            7.58967082947862858866017177206231e-02 7.96770050912385174868290960148443e-01 3.45528806082068219371339345968863e-03
            3.13633037996470243058411142555997e-02 8.35167147087011674244649839238264e-01 2.37825419634513481395576839361183e-03
            6.00374098975725578597462117613759e-03 8.57032386971469550829283434723038e-01 1.06662031788896114262166214814442e-03
            9.93996259010242688702874147566035e-01 4.71585501322903648951934130195696e-03 7.67475790123426755359476586493273e-06
            9.68636696200352975694158885744400e-01 2.46354387551566134106906247325242e-02 9.17353649372818115296027152716363e-05
            9.24103291705213436557642125990242e-01 5.96158083618486961396953915937047e-02 3.38068196908658990997714743897973e-04
            8.62208865680085256855136321973987e-01 1.08233018824375804012305479773204e-01 7.99405798997733312537261607388928e-04
            7.85486086304269637281549876206554e-01 1.68497694526671154546804132223770e-01 1.48258335465236519221110533806041e-03
            6.97075673538781748206361044140067e-01 2.37942843638379225579271292190242e-01 2.34409217545607923005590578213742e-03
            6.00597046998717187804572859022301e-01 3.13725462411345801250206477561733e-01 3.29437994774776834322471685823075e-03
            5.00000000000000000000000000000000e-01 3.92743043152134763129623706845450e-01 4.21031542509575380101027874957254e-03
            3.99402953001282645661973447204218e-01 4.71760623892924058075948323676130e-01 4.95388142085857914598268081363130e-03
            3.02924326461218529349395112149068e-01 5.47543242665890272924400505871745e-01 5.39411823121517097073818192143335e-03
            2.14513913695730473740752586309100e-01 6.16988391777598454979170128353871e-01 5.42877884610113315561052971247591e-03
            1.37791134319915020700619834315148e-01 6.77253067479893666735790702659870e-01 5.00217064453254388334269009419586e-03
            7.58967082947862858866017177206231e-02 7.25870277942421093797520370571874e-01 4.11625142385254206428557566255222e-03
            3.13633037996470243058411142555997e-02 7.60850647549113023870859251474030e-01 2.83319134314485854275744713959284e-03
            6.00374098975725578597462117613759e-03 7.80770231291040617271903556684265e-01 1.27065452284684000938685954906759e-03
            9.93996259010242688702874147566035e-01 4.18506179418750862836207105033282e-03 8.59293576793571239349177337540908e-06
            9.68636696200352975694158885744400e-01 2.18625961205404194254153793508522e-02 1.02710223397070615119000547288408e-04
            9.24103291705213436557642125990242e-01 5.29057490539647859439931210090435e-02 3.78513347079099699444421078808887e-04
            8.62208865680085256855136321973987e-01 9.60508477637273538896067748282803e-02 8.95043566416359008541814290538241e-04
            7.85486086304269637281549876206554e-01 1.49532430872891392015233691381582e-01 1.65995379933608043498960427086786e-03
            6.97075673538781748206361044140067e-01 2.11161178899235524308863887199550e-01 2.62453014896735789887594059166531e-03
            6.00597046998717187804572859022301e-01 2.78414082476747593020149906806182e-01 3.68850661486264932639733693520157e-03
            5.00000000000000000000000000000000e-01 3.48537836769390818592029290812206e-01 4.71402101228214458206178250065932e-03
            3.99402953001282645661973447204218e-01 4.18661591062034321719664831107366e-01 5.54654432090450049547269983918341e-03
            3.02924326461218529349395112149068e-01 4.85914494639546057364043463167036e-01 6.03944933273122073341676596669458e-03
            2.14513913695730473740752586309100e-01 5.47543242665890383946702968387399e-01 6.07825660733519578665662308480933e-03
            1.37791134319915020700619834315148e-01 6.01024825775054227783300575538306e-01 5.60061067747936146277165647688889e-03
            7.58967082947862858866017177206231e-02 6.44169924484817135734715520811733e-01 4.60870356368510772360203020525660e-03
            3.13633037996470243058411142555997e-02 6.75213077418241325311498712835601e-01 3.17214321848510727441805556736654e-03
            6.00374098975725578597462117613759e-03 6.92890611744594320242640606011264e-01 1.42267063516152625010646737280240e-03
            9.93996259010242688702874147566035e-01 3.60582910939343014369851125877631e-03 9.15932449658844526985471135027694e-06
            9.68636696200352975694158885744400e-01 1.88367076461916811702668184125287e-02 1.09480192872064160621893569302898e-04
            9.24103291705213436557642125990242e-01 4.55833388787718549184369010163209e-02 4.03462409799921583979098738126368e-04
            8.62208865680085256855136321973987e-01 8.27569483751444262553675912386097e-02 9.54038838970706535268229142587870e-04
            7.85486086304269637281549876206554e-01 1.28836423105793351773229460377479e-01 1.76936682736504417856604476355642e-03
            6.97075673538781748206361044140067e-01 1.81935455936683060373937337317329e-01 2.79752158455229922237594841760711e-03
            6.00597046998717187804572859022301e-01 2.39880234135137898610068418747687e-01 3.93162824740350604763028385946200e-03
            5.00000000000000000000000000000000e-01 3.00298523499358538391135198253323e-01 5.02473768003051327740715592540255e-03
            3.99402953001282645661973447204218e-01 3.60716812863579427972382518419181e-01 5.91213534487741894374313034177248e-03
            3.02924326461218529349395112149068e-01 4.18661591062033988652757443560404e-01 6.43752935121481937447152077425017e-03
            2.14513913695730473740752586309100e-01 4.71760623892923780520192167386995e-01 6.47889454124130331064135290830563e-03
            1.37791134319915020700619834315148e-01 5.17840098623572719915841844340321e-01 5.96976539327895294739789733284852e-03
            7.58967082947862858866017177206231e-02 5.55013708119945436969544516614405e-01 4.91247841114914388094270236706507e-03
            3.13633037996470243058411142555997e-02 5.81760339352525579492692031635670e-01 3.38122963704375360458054267098760e-03
            6.00374098975725578597462117613759e-03 5.96991217889323744216767408943269e-01 1.51644354748183864123323161265944e-03
            9.93996259010242688702874147566035e-01 3.00187049487868340413854184589582e-03 9.35073307890573744195501088016798e-06
            9.68636696200352975694158885744400e-01 1.56816518998235399084961727567133e-02 1.11768074311057922067338954441595e-04
            9.24103291705213436557642125990242e-01 3.79483541473932817211789370048791e-02 4.11893835928222402844450789061170e-04
            8.62208865680085256855136321973987e-01 6.88955671599573993280074546419200e-02 9.73976032124092001678039576262336e-04
            7.85486086304269637281549876206554e-01 1.07256956847865209114800677525636e-01 1.80634247946158616252021467118993e-03
            6.97075673538781748206361044140067e-01 1.51462163230609125896819477929967e-01 2.85598328013919816950605401473240e-03
            6.00597046998717187804572859022301e-01 1.99701476500641406097713570488850e-01 4.01379013492200471119319971080586e-03
            5.00000000000000000000000000000000e-01 2.49999999999999972244424384371086e-01 5.12974301270642790012699308022093e-03
            3.99402953001282645661973447204218e-01 3.00298523499358704924588892026804e-01 6.03568522513905220128194173412339e-03
            3.02924326461218529349395112149068e-01 3.48537836769390763080878059554379e-01 6.57205874442508795885142802717382e-03
            2.14513913695730473740752586309100e-01 3.92743043152134763129623706845450e-01 6.61428837072994615930987549745623e-03
            1.37791134319915020700619834315148e-01 4.31104432840042517405265698471339e-01 6.09451960753568805495561377938429e-03
            7.58967082947862858866017177206231e-02 4.62051645852606829301123525510775e-01 5.01513778616010513777823121017718e-03
            3.13633037996470243058411142555997e-02 4.84318348100176487847079442872200e-01 3.45188947353680707433309926557286e-03
            6.00374098975725578597462117613759e-03 4.96998129505121344351437073783018e-01 1.54813369119230753029325953207263e-03
            9.93996259010242688702874147566035e-01 2.39791188036393579721683444461178e-03 9.15932449658848592743617955669322e-06
            9.68636696200352975694158885744400e-01 1.25265961534553917078316231936697e-02 1.09480192872064634960344031711088e-04
            9.24103291705213436557642125990242e-01 3.03133694160146981155801171325948e-02 4.03462409799923372912683339208684e-04
            8.62208865680085256855136321973987e-01 5.50341859447703515839656063235452e-02 9.54038838970710763656701836055163e-04
            7.85486086304269637281549876206554e-01 8.56774905899370109452206634159666e-02 1.76936682736505198482168665918834e-03
            6.97075673538781748206361044140067e-01 1.20988870524535149786338195099233e-01 2.79752158455231136544028025525677e-03
            6.00597046998717187804572859022301e-01 1.59522718866144858074207490972185e-01 3.93162824740352339486504362753294e-03
            5.00000000000000000000000000000000e-01 1.99701476500641295075411107973196e-01 5.02473768003053496145060563549123e-03
            3.99402953001282645661973447204218e-01 2.39880234135137870854492803118774e-01 5.91213534487744409723353200547535e-03
            3.02924326461218529349395112149068e-01 2.78414082476747426486696213032701e-01 6.43752935121484799740887439156722e-03
            2.14513913695730473740752586309100e-01 3.13725462411345579205601552530425e-01 6.47889454124133193357870652562269e-03
            1.37791134319915020700619834315148e-01 3.44368767056512203872387090086704e-01 5.96976539327897896825003698495493e-03
            7.58967082947862858866017177206231e-02 3.69089583585268055099248840633663e-01 4.91247841114916469762441408875020e-03
            3.13633037996470243058411142555997e-02 3.86876356847827285179164391593076e-01 3.38122963704376834973008847384790e-03
            6.00374098975725578597462117613759e-03 3.97005041120918722441501813591458e-01 1.51644354748184536328670102278693e-03
            9.93996259010242688702874147566035e-01 1.81867919556985991463848861826591e-03 8.59293576793570900535998435820773e-06
            9.68636696200352975694158885744400e-01 9.50070767910666733047087006980291e-03 1.02710223397070560908891923013186e-04
            9.24103291705213436557642125990242e-01 2.29909592408217983150464647223998e-02 3.78513347079099536814095205983222e-04
            8.62208865680085256855136321973987e-01 4.17402865561874863997715578989300e-02 8.95043566416358574860945296336467e-04
            7.85486086304269637281549876206554e-01 6.49814828228390956033067027419747e-02 1.65995379933607956762786628246431e-03
            6.97075673538781748206361044140067e-01 9.17631475619828107515019155471236e-02 2.62453014896735659783333360905999e-03
            6.00597046998717187804572859022301e-01 1.20988870524535316319791888872714e-01 3.68850661486264802535472995259624e-03
            5.00000000000000000000000000000000e-01 1.51462163230609236919121940445621e-01 4.71402101228214284733830652385222e-03
            3.99402953001282645661973447204218e-01 1.81935455936683282418542262348637e-01 5.54654432090449789338748587397276e-03
            3.02924326461218529349395112149068e-01 2.11161178899235635331166349715204e-01 6.03944933273121726396981401308039e-03
            2.14513913695730473740752586309100e-01 2.37942843638379419868300601592637e-01 6.07825660733519318457140911959868e-03
            1.37791134319915020700619834315148e-01 2.61184039905030973560684515177854e-01 5.60061067747935886068644251167825e-03
            7.58967082947862858866017177206231e-02 2.79933367220396800423287686498952e-01 4.60870356368510598887855422844950e-03
            3.13633037996470243058411142555997e-02 2.93423618782111872427265097940108e-01 3.17214321848510597337544858476122e-03
            6.00374098975725578597462117613759e-03 3.01105647265648701527140929101733e-01 1.42267063516152559958516388149974e-03
            9.93996259010242688702874147566035e-01 1.28788597652833118611948037823822e-03 7.67475790123429804678086701974493e-06
            9.68636696200352975694158885744400e-01 6.72786504449046900838693474611318e-03 9.17353649372821638953087730605773e-05
            9.24103291705213436557642125990242e-01 1.62808999329378777110033382768961e-02 3.38068196908660292040321726503294e-04
            8.62208865680085256855136321973987e-01 2.95581154955390085214972373250930e-02 7.99405798997736456723561815351786e-04
            7.85486086304269637281549876206554e-01 4.60162191690592914383728384564165e-02 1.48258335465237083006240226268346e-03
            6.97075673538781748206361044140067e-01 6.49814828228390539699432792986045e-02 2.34409217545608833735415466037466e-03
            6.00597046998717187804572859022301e-01 8.56774905899370803341597024882503e-02 3.29437994774778091996991769008218e-03
            5.00000000000000000000000000000000e-01 1.07256956847865222992588485340093e-01 4.21031542509576941352156254083638e-03
            3.99402953001282645661973447204218e-01 1.28836423105793435039956307264220e-01 4.95388142085859822794091655850934e-03
            3.02924326461218529349395112149068e-01 1.49532430872891364259658075752668e-01 5.39411823121519178741989364311848e-03
            2.14513913695730473740752586309100e-01 1.68497694526671182302379747852683e-01 5.42877884610115397229224143416104e-03
            1.37791134319915020700619834315148e-01 1.84955798200191423585891925540636e-01 5.00217064453256296530092583907390e-03
            7.58967082947862858866017177206231e-02 1.98233013762792648071453527336416e-01 4.11625142385255767679685945381607e-03
            3.13633037996470243058411142555997e-02 2.07786048651240007334450865528197e-01 2.83319134314486938477917199463718e-03
            6.00374098975725578597462117613759e-03 2.13226027719202154697697437768511e-01 1.27065452284684499671685298238799e-03
            9.93996259010242688702874147566035e-01 8.27262281141636929976401759745386e-04 6.44239056733915846863179621717421e-06
            9.68636696200352975694158885744400e-01 4.32158520657347205301546111400057e-03 7.70050413796524838060467144096322e-05
            9.24103291705213436557642125990242e-01 1.04578935270863444123579100164534e-02 2.83783200839656120882670276017734e-04
            8.62208865680085256855136321973987e-01 1.89863966971688322149525163240469e-02 6.71041933206906106995637273371358e-04
            7.85486086304269637281549876206554e-01 2.95581154955390501548606607684633e-02 1.24451886850663706422792653683018e-03
            6.97075673538781748206361044140067e-01 4.17402865561875141553471735278436e-02 1.96769168675706542306902413486114e-03
            6.00597046998717187804572859022301e-01 5.50341859447704695451619727464276e-02 2.76538785636329473988093141656464e-03
            5.00000000000000000000000000000000e-01 6.88955671599574964725221093431173e-02 3.53424781982989094988867329050208e-03
            3.99402953001282645661973447204218e-01 8.27569483751445789110334771976341e-02 4.15841637588638696892973101171265e-03
            3.02924326461218529349395112149068e-01 9.60508477637274787896970451583911e-02 4.52796255713865374720183964996068e-03
            2.14513913695730473740752586309100e-01 1.08233018824375970545759173546685e-01 4.55705757502365953887091265528397e-03
            1.37791134319915020700619834315148e-01 1.18804737622746167668985606269416e-01 4.19895160098465985293314517434737e-03
            7.58967082947862858866017177206231e-02 1.27333240792828705778561015904415e-01 3.45528806082068653052208340170637e-03
            3.13633037996470243058411142555997e-02 1.33469549113341551249689587166358e-01 2.37825419634513741604098235882248e-03
            6.00374098975725578597462117613759e-03 1.36963872038773387673771253503219e-01 1.06662031788896244366426913074974e-03
            9.93996259010242688702874147566035e-01 4.55664178577066376028825711586023e-04 4.94632226051369185555435861734175e-06
            9.68636696200352975694158885744400e-01 2.38037151964257684758075050979187e-03 5.91227350106575901445178644610934e-05
            9.24103291705213436557642125990242e-01 5.76031032998390243871078553183906e-03 2.17882344884403559431840236015887e-04
            8.62208865680085256855136321973987e-01 1.04578935270862889012066787586264e-02 5.15210870447172560732118640203225e-04
            7.85486086304269637281549876206554e-01 1.62808999329378117915112511582265e-02 9.55513534701166221754786711528595e-04
            6.97075673538781748206361044140067e-01 2.29909592408216907621909541603600e-02 1.51074932360924403529356396802541e-03
            6.00597046998717187804572859022301e-01 3.03133694160145975016185104777833e-02 2.12320246186711901817401049186174e-03
            5.00000000000000000000000000000000e-01 3.79483541473931360044069549530832e-02 2.71351581104416162901205034074792e-03
            3.99402953001282645661973447204218e-01 4.55833388787717022627710150572966e-02 3.19273835908195783087060171112626e-03
            3.02924326461218529349395112149068e-01 5.29057490539645777771760037921922e-02 3.47646758715495747885104727004091e-03
            2.14513913695730473740752586309100e-01 5.96158083618484671561965626551682e-02 3.49880608606004421187729569453495e-03
            1.37791134319915020700619834315148e-01 6.54388147676999848423307071243471e-02 3.22386039121316586228416234405358e-03
            7.58967082947862858866017177206231e-02 7.01363979648023938562317880496266e-02 2.65289228789857044546462994105696e-03
            3.13633037996470243058411142555997e-02 7.35163367751437146768722641354543e-02 1.82596996403474351282347765135228e-03
            6.00374098975725578597462117613759e-03 7.54410441162092215705570197314955e-02 8.18927037525047306874836916534832e-04
            9.93996259010242688702874147566035e-01 1.88297152596153815088883543005238e-04 3.24799999064399304578407319532563e-06
            9.68636696200352975694158885744400e-01 9.83656825228955119036489129769052e-04 3.88229138029352934132452934701263e-05
            9.24103291705213436557642125990242e-01 2.38037151964258118438944045180961e-03 1.43072330688081574522063643684078e-04
            8.62208865680085256855136321973987e-01 4.32158520657345730786591531114027e-03 3.38312955415547738421272461195599e-04
            7.85486086304269637281549876206554e-01 6.72786504449045599796086492005998e-03 6.27437475423869113130836971947701e-04
            6.97075673538781748206361044140067e-01 9.50070767910664130961873041769650e-03 9.92032773141360941712707521844550e-04
            6.00597046998717187804572859022301e-01 1.25265961534553743605968634255987e-02 1.39419981413898327188283232658250e-03
            5.00000000000000000000000000000000e-01 1.56816518998235086834736051741856e-02 1.78182877392393215938704020828709e-03
            3.99402953001282645661973447204218e-01 1.88367076461916568841381547372293e-02 2.09651001577684402074619463007821e-03
            3.02924326461218529349395112149068e-01 2.18625961205403743226050039538677e-02 2.28282066874081540283625280096658e-03
            2.14513913695730473740752586309100e-01 2.46354387551565683078802493355397e-02 2.29748923265827957596041386523211e-03
            1.37791134319915020700619834315148e-01 2.70417185930735617938047710140381e-02 2.11694628230924128051149146756416e-03
            7.58967082947862858866017177206231e-02 2.89829322800044496266647087168167e-02 1.74202036835732093113249252525065e-03
            3.13633037996470243058411142555997e-02 3.03796469744180702710067976113351e-02 1.19902224597172356659879888240994e-03
            6.00374098975725578597462117613759e-03 3.11750066470508728488919558685666e-02 5.37748021687375855545698577486746e-04
            9.93996259010242688702874147566035e-01 3.60449058720920965850215145209745e-05 1.41952736129656462027589201857491e-06
            9.68636696200352975694158885744400e-01 1.88297152596150670902583335042380e-04 1.69674225822881880607409521566353e-05
            9.24103291705213436557642125990242e-01 4.55664178577059599765247677183311e-04 6.25292760594908428707985881267462e-05
            8.62208865680085256855136321973987e-01 8.27262281141620233262945482977102e-04 1.47858527794593653690788381815935e-04
            7.85486086304269637281549876206554e-01 1.28788597652830711683125120003979e-03 2.74219417005117582167988965480276e-04
            6.97075673538781748206361044140067e-01 1.81867919556982456964766559082136e-03 4.33564553212285278428056445321204e-04
            6.00597046998717187804572859022301e-01 2.39791188036389242912993502443442e-03 6.09330292175421818490210768004545e-04
            5.00000000000000000000000000000000e-01 3.00187049487862745930644159386702e-03 7.78742212135606409403876249086807e-04
            3.99402953001282645661973447204218e-01 3.60582910939336465788729313430849e-03 9.16272579803009625150356232126114e-04
            3.02924326461218529349395112149068e-01 4.18506179418742969844391410561002e-03 9.97699017717175928518513750020702e-04
            2.14513913695730473740752586309100e-01 4.71585501322894801862206648479514e-03 1.00410986374296039291831306172753e-03
            1.37791134319915020700619834315148e-01 5.17647870861563490219037220185783e-03 9.25204180661707599464538276379244e-04
            7.58967082947862858866017177206231e-02 5.54807681118019754146208910583482e-03 7.61344083726070734383095661712559e-04
            3.13633037996470243058411142555997e-02 5.81544383716110519638720077750804e-03 5.24028599095731633002170912760675e-04
            6.00374098975725578597462117613759e-03 5.96769608388516420438563159223122e-03 2.35020945956645923701627820534554e-04
            ];
        
    case  29
        
        % ALG. DEG.:   29
        % PTS CARD.:  256
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.1e-15
        
        xyw=[
            9.94700467495824969255124869960127e-01 5.27144745941222295920747242803373e-03 9.76778103010145115178328688310128e-07
            9.72287511536616300134028278989717e-01 2.75656252300003658251004168278087e-02 1.17108561784217877176084154910995e-05
            9.32815601193915933109224170038942e-01 6.68283529008377730207257627625950e-02 4.33975545762342102733624682464608e-05
            8.77702204177501554838158881466370e-01 1.21649674678348099221558697990986e-01 1.03463418054530522922733470903012e-04
            8.08938122201322107329701793787535e-01 1.90049339166975139736592836925411e-01 1.94018513284900542912453369481796e-04
            7.29008388828613629328856404754333e-01 2.69555482319624839515626035790774e-01 3.11167169456441714686656396793296e-04
            6.40801775389629346690867350844201e-01 3.57294641943606106071484873609734e-01 4.45238036373906689250473656116469e-04
            5.47506254918818435761806995287770e-01 4.50095739771187941524743791887886e-01 5.81912613580765265310079659144549e-04
            4.52493745081181064637831923391786e-01 5.44604727724637527330742159392685e-01 7.04099889147563975412558168187616e-04
            3.59198224610370542286830186640145e-01 6.37405825552219029717093690123875e-01 7.94294917489840471627737805704328e-04
            2.70991611171386148626538670214359e-01 7.25144985176200296272952527942834e-01 8.37086712320163155685859290855433e-04
            1.91061877798678114714903131243773e-01 8.04651128328849662985078339261236e-01 8.21456240340928613326287877072218e-04
            1.22297795822498389650689887275803e-01 8.73050792817476772889051517267944e-01 7.42532352831611324600613599500321e-04
            6.71843988060842889353807549923658e-02 9.27872114594986974189794182166224e-01 6.02549351959222908962920950415310e-04
            2.77124884633839219105766460415907e-02 9.67134842265824357099290864425711e-01 4.10873214317274289796022568310718e-04
            5.29953250417514176717759255552664e-03 9.89429020036412643079870576912072e-01 1.83337234923728551730034985745021e-04
            9.94700467495824969255124869960127e-01 5.15266927079175308251679510362919e-03 2.23949801733855506137311347503172e-06
            9.72287511536616300134028278989717e-01 2.69445064465504702244658119525411e-02 2.68499458700909916135726729358169e-05
            9.32815601193915933109224170038942e-01 6.53225519292510992119815682599437e-02 9.94992999242211913647065957277960e-05
            8.77702204177501554838158881466370e-01 1.18908619566670148426368314176216e-01 2.37214694807490563592880516985417e-04
            8.08938122201322107329701793787535e-01 1.85767077714389661213800764016923e-01 4.44833964325668310135775129765534e-04
            7.29008388828613629328856404754333e-01 2.63481759273125559772665837954264e-01 7.13425349023526938520289863987500e-04
            6.40801775389629346690867350844201e-01 3.49243947954787914422780659151613e-01 1.02081495953919754345051718047444e-03
            5.47506254918818435761806995287770e-01 4.39954017390866036762986368557904e-01 1.33417420022250961600396301776073e-03
            4.52493745081181064637831923391786e-01 5.32333494145750818482554223010084e-01 1.61431782806651259037922230277218e-03
            3.59198224610370542286830186640145e-01 6.23043563581828552244701313611586e-01 1.82111155790529968996616094756291e-03
            2.70991611171386148626538670214359e-01 7.08805752263490962405967366066761e-01 1.91922200836025904149839682588663e-03
            1.91061877798678114714903131243773e-01 7.86520433822226472386773821199313e-01 1.88338540340393569699761755487089e-03
            1.22297795822498389650689887275803e-01 8.53378891969946096196508733555675e-01 1.70243346656886963483290564624895e-03
            6.71843988060842889353807549923658e-02 9.06964959607365006633017401327379e-01 1.38148887132382358275406453174128e-03
            2.77124884633839219105766460415907e-02 9.45343005090065524598230695119128e-01 9.42025364824847937818275145360758e-04
            5.29953250417514176717759255552664e-03 9.67134842265824468121593326941365e-01 4.20344572478360031363847637564390e-04
            9.94700467495824969255124869960127e-01 4.94348659892872993593115893418144e-03 3.42321662715594550604157608741929e-06
            9.72287511536616300134028278989717e-01 2.58506415865506732065792760977274e-02 4.10418676101188421639562320386574e-05
            9.32815601193915933109224170038942e-01 6.26706553631491136702180710926768e-02 1.52091073648617184504994170879399e-04
            8.77702204177501554838158881466370e-01 1.14081291934854611724858841625974e-01 3.62597904165934111029440112261568e-04
            8.08938122201322107329701793787535e-01 1.78225500404012277000376229807443e-01 6.79957299008012274257883866823704e-04
            7.29008388828613629328856404754333e-01 2.52785202693344701607713886915008e-01 1.09051648990259931505664159345770e-03
            6.40801775389629346690867350844201e-01 3.35065707837710224925586999233929e-01 1.56038126209058838900212151656888e-03
            5.47506254918818435761806995287770e-01 4.22093224854388937572480244853068e-01 2.03937099759161867204571016998216e-03
            4.52493745081181064637831923391786e-01 5.10722376339527550648256237764144e-01 2.46758853446939424927664141762307e-03
            3.59198224610370542286830186640145e-01 5.97749893356205874717090864578495e-01 2.78368603886326162685782570349602e-03
            2.70991611171386148626538670214359e-01 6.80030398500571453546115208155243e-01 2.93365416685218826503556144302820e-03
            1.91061877798678114714903131243773e-01 7.54590100789903517330969862086931e-01 2.87887561335603689091633405894299e-03
            1.22297795822498389650689887275803e-01 8.18734309259061210362062865897315e-01 2.60227895013325893208699923775384e-03
            6.71843988060842889353807549923658e-02 8.70144945830766625149976789543871e-01 2.11169451275816395102724598586974e-03
            2.77124884633839219105766460415907e-02 9.06964959607365006633017401327379e-01 1.43994630363790333989204572873177e-03
            5.29953250417514176717759255552664e-03 9.27872114594987085212096644681878e-01 6.42523689908295193677634227213957e-04
            9.94700467495824969255124869960127e-01 4.65141136002473858102046477824842e-03 4.48338208620572117375129364558184e-06
            9.72287511536616300134028278989717e-01 2.43233122075554065999458686064827e-02 5.37524772951656875231811472115595e-05
            9.32815601193915933109224170038942e-01 5.89678949184402889649447843112284e-02 1.99193468990165051127685935661304e-04
            8.77702204177501554838158881466370e-01 1.07341044959456871632497154678276e-01 4.74893974029307129169213963493235e-04
            8.08938122201322107329701793787535e-01 1.67695431278192086388401094154688e-01 8.90539134910106936447726777572598e-04
            7.29008388828613629328856404754333e-01 2.37849934438738280784164658143709e-01 1.42824793989250724973727457722816e-03
            6.40801775389629346690867350844201e-01 3.15269073477167649510732871931395e-01 2.04362918274331086671002921661966e-03
            5.47506254918818435761806995287770e-01 3.97154757434285576866983547006384e-01 2.67096137743585157822701070529092e-03
            4.52493745081181064637831923391786e-01 4.80547446743216477571536415780429e-01 3.23179729375120457918191263502194e-03
            3.59198224610370542286830186640145e-01 5.62433130700334071860879703308456e-01 3.64578975845572918548787022530178e-03
            2.70991611171386148626538670214359e-01 6.39852269738763523854174763982883e-01 3.84220280844899553934768121621346e-03
            1.91061877798678114714903131243773e-01 7.10006772899309357427455324796028e-01 3.77045941263096616447025866136755e-03
            1.22297795822498389650689887275803e-01 7.70361159218044599938934879901353e-01 3.40820114502388225169449142981648e-03
            6.71843988060842889353807549923658e-02 8.18734309259061099339760403381661e-01 2.76568338530904607105997961014054e-03
            2.77124884633839219105766460415907e-02 8.53378891969945874151903808524366e-01 1.88589568408117653347533249075241e-03
            5.29953250417514176717759255552664e-03 8.73050792817476772889051517267944e-01 8.41512388799933318435375628752126e-04
            9.94700467495824969255124869960127e-01 4.28699387247221969793331552978088e-03 5.38154146400920609945956948894086e-06
            9.72287511536616300134028278989717e-01 2.24176883790953677955126721599299e-02 6.45207523684329344515128723536179e-05
            9.32815601193915933109224170038942e-01 5.43480214114183887708087183909811e-02 2.39098049668483607008151459183409e-04
            8.77702204177501554838158881466370e-01 9.89313493020125417354293517746555e-02 5.70029848696138974166458535108859e-04
            8.08938122201322107329701793787535e-01 1.54557236650721019355358976099524e-01 1.06894152398630173234073126309340e-03
            7.29008388828613629328856404754333e-01 2.19215445073292103383977291741758e-01 1.71436994697948060124848002061526e-03
            6.40801775389629346690867350844201e-01 2.90569137314362013757573777184007e-01 2.45303098699309329866480311466148e-03
            5.47506254918818435761806995287770e-01 3.66039440453814757248096611874644e-01 3.20603712221252494518619613472765e-03
            4.52493745081181064637831923391786e-01 4.42898681747507794170815031975508e-01 3.87922572851998699247633162201510e-03
            3.59198224610370542286830186640145e-01 5.18368984886960260105581710377010e-01 4.37615362173915389831790889729746e-03
            2.70991611171386148626538670214359e-01 5.89722677128030170479178195819259e-01 4.61191424893696213377358716911658e-03
            1.91061877798678114714903131243773e-01 6.54380885550600921440889123914531e-01 4.52579844351599317603396244180658e-03
            1.22297795822498389650689887275803e-01 7.10006772899309468449757787311682e-01 4.09096869884492675417675400240114e-03
            6.71843988060842889353807549923658e-02 7.54590100789903517330969862086931e-01 3.31973486269558160435821925204891e-03
            2.77124884633839219105766460415907e-02 7.86520433822226472386773821199313e-01 2.26369861536114590563895099251113e-03
            5.29953250417514176717759255552664e-03 8.04651128328849774007380801776890e-01 1.01009321216181512798071295833324e-03
            9.94700467495824969255124869960127e-01 3.86340365241350739561321248061176e-03 6.08520876952028885742158187399653e-06
            9.72287511536616300134028278989717e-01 2.02026365651228542730244441827381e-02 7.29572095196558291886959612959629e-05
            9.32815601193915933109224170038942e-01 4.89779903280423770106644099087134e-02 2.70361486267155714472570915418714e-04
            8.77702204177501554838158881466370e-01 8.91561190898502997637109501738450e-02 6.44564509513204878660141972090969e-04
            8.08938122201322107329701793787535e-01 1.39285711700583680894283133966383e-01 1.20871173795989881802337428950977e-03
            7.29008388828613629328856404754333e-01 1.97555157846122514397890768123034e-01 1.93853361631252519951817969712238e-03
            6.40801775389629346690867350844201e-01 2.61858518993304811850464375311276e-01 2.77377880924747736990365609699438e-03
            5.47506254918818435761806995287770e-01 3.29871736056657594016172652118257e-01 3.62524480057817765840666446308660e-03
            4.52493745081181064637831923391786e-01 3.99136652771956423890742371440865e-01 4.38645666488135695104677580502539e-03
            3.59198224610370542286830186640145e-01 4.67149869835308928500694491958711e-01 4.94836072041263726889459562130469e-03
            2.70991611171386148626538670214359e-01 5.31453230982491309219994946033694e-01 5.21494839714548540088312478246735e-03
            1.91061877798678114714903131243773e-01 5.89722677128029837412270808272297e-01 5.11757245795657102205300503783292e-03
            1.22297795822498389650689887275803e-01 6.39852269738763301809569838951575e-01 4.62588623882831222189349063000918e-03
            6.71843988060842889353807549923658e-02 6.80030398500571120479207820608281e-01 3.75380917048766769850143276698873e-03
            2.77124884633839219105766460415907e-02 7.08805752263490518316757516004145e-01 2.55969014786411319661851848650258e-03
            5.29953250417514176717759255552664e-03 7.25144985176200074228347602911526e-01 1.14216867300708413597576385001275e-03
            9.94700467495824969255124869960127e-01 3.39594983741040804675614950269846e-03 6.56894517899701801797894204493211e-06
            9.72287511536616300134028278989717e-01 1.77582118078008603656581243512846e-02 7.87568558284646992552133215603760e-05
            9.32815601193915933109224170038942e-01 4.30518820334235621793972370596748e-02 2.91853549987752781350730524323467e-04
            8.77702204177501554838158881466370e-01 7.83686446892953580967144944224856e-02 6.95803396019415332403312834941289e-04
            8.08938122201322107329701793787535e-01 1.22432790502669239240951526426215e-01 1.30479683517821347382703578432483e-03
            7.29008388828613629328856404754333e-01 1.73651905554320501456899705772230e-01 2.09263503283284692593757547740552e-03
            6.40801775389629346690867350844201e-01 2.30174860047128393425452941301046e-01 2.99427704565782609388846502440629e-03
            5.47506254918818435761806995287770e-01 2.89958795200723484786209382946254e-01 3.91342930989027355109799444221608e-03
            4.52493745081181064637831923391786e-01 3.50842980188906194971565355444909e-01 4.73515280297001715098526020142344e-03
            3.59198224610370542286830186640145e-01 4.10626915342501064287716872058809e-01 5.34172475085933472910282304724205e-03
            2.70991611171386148626538670214359e-01 4.67149869835308984011845723216538e-01 5.62950449682724646249809552500665e-03
            1.91061877798678114714903131243773e-01 5.18368984886960038060976785345702e-01 5.52438777355404499885827007688022e-03
            1.22297795822498389650689887275803e-01 5.62433130700333960838577240792802e-01 4.99361554517962526189567284973236e-03
            6.71843988060842889353807549923658e-02 5.97749893356205652672485939547187e-01 4.05221375096612330957146497212307e-03
            2.77124884633839219105766460415907e-02 6.23043563581828330200096388580278e-01 2.76316966161600655396690662257697e-03
            5.29953250417514176717759255552664e-03 6.37405825552218918694791227608221e-01 1.23296400868474608641944811893154e-03
            9.94700467495824969255124869960127e-01 2.90152719418141843976677307637146e-03 6.81526505898426486447028507797974e-06
            9.72287511536616300134028278989717e-01 1.51727607730681411657425172734293e-02 8.17100513183382121422046462377864e-05
            9.32815601193915933109224170038942e-01 3.67838785792914263472752622874395e-02 3.02797366604869340467320881771229e-04
            8.77702204177501554838158881466370e-01 6.69588081756024161572682373844145e-02 7.21894374758316986588624342147114e-04
            8.08938122201322107329701793787535e-01 1.04607573171311099557989621189336e-01 1.35372362495816880162413653465592e-03
            7.29008388828613629328856404754333e-01 1.48369602146862400449123242651694e-01 2.17110388834903774227247374994931e-03
            6.40801775389629346690867350844201e-01 1.96663274729912629368300258647650e-01 3.10655534033642743638181649146190e-03
            5.47506254918818435761806995287770e-01 2.47743155743588239214858504055883e-01 4.06017363667089854761815814754300e-03
            4.52493745081181064637831923391786e-01 2.99763099175230474102704647521023e-01 4.91270981377870535888874314878194e-03
            3.59198224610370542286830186640145e-01 3.50842980188905917415809199155774e-01 5.54202677252386630146752111159003e-03
            2.70991611171386148626538670214359e-01 3.99136652771956146334986215151730e-01 5.84059757711047691786099278488109e-03
            1.91061877798678114714903131243773e-01 4.42898681747507239059302719397238e-01 5.73153922577435658014533004234181e-03
            1.22297795822498389650689887275803e-01 4.80547446743215977971175334459986e-01 5.18086429642874123080442672062418e-03
            6.71843988060842889353807549923658e-02 5.10722376339526884514441462670220e-01 4.20416216545617307115412231155460e-03
            2.77124884633839219105766460415907e-02 5.32333494145750152348739447916159e-01 2.86678197697065812246663973894556e-03
            5.29953250417514176717759255552664e-03 5.44604727724636972219229846814414e-01 1.27919723766934116844506963417416e-03
            9.94700467495824969255124869960127e-01 2.39800530999360970302314299829050e-03 6.81526505898424961787723450057364e-06
            9.72287511536616300134028278989717e-01 1.25397276903154893112901646645696e-02 8.17100513183380359593516173433159e-05
            9.32815601193915933109224170038942e-01 3.04005202267926093184780000910905e-02 3.02797366604868635735908766193347e-04
            8.77702204177501554838158881466370e-01 5.53389876468959179822704186335613e-02 7.21894374758315360285365613890463e-04
            8.08938122201322107329701793787535e-01 8.64543046273667514789451615797589e-02 1.35372362495816576585805357524350e-03
            7.29008388828613629328856404754333e-01 1.22622009024523845321930082263862e-01 2.17110388834903297178291481372980e-03
            6.40801775389629346690867350844201e-01 1.62534949880457912918529927992495e-01 3.10655534033642049748791258423353e-03
            5.47506254918818435761806995287770e-01 2.04750589337593102978729575625039e-01 4.06017363667088900663904027510398e-03
            4.52493745081181064637831923391786e-01 2.47743155743588211459282888426969e-01 4.91270981377869408318614929953583e-03
            3.59198224610370542286830186640145e-01 2.89958795200723262741604457914946e-01 5.54202677252385415840318927394037e-03
            2.70991611171386148626538670214359e-01 3.29871736056657316460416495829122e-01 5.84059757711046390743492295882788e-03
            1.91061877798678114714903131243773e-01 3.66039440453814257647735530554201e-01 5.73153922577434356971926021628860e-03
            1.22297795822498389650689887275803e-01 3.97154757434285132777773696943768e-01 5.18086429642872995510183287137806e-03
            6.71843988060842889353807549923658e-02 4.22093224854388382460967932274798e-01 4.20416216545616353017500443911558e-03
            2.77124884633839219105766460415907e-02 4.39954017390865426140322824721807e-01 2.86678197697065161725360482591896e-03
            5.29953250417514176717759255552664e-03 4.50095739771187386413231479309616e-01 1.27919723766933856635985566896352e-03
            9.94700467495824969255124869960127e-01 1.90358266676462226443811154297236e-03 6.56894517899701293578125851913008e-06
            9.72287511536616300134028278989717e-01 9.95427665558278051971541344755678e-03 7.87568558284646450451046972851543e-05
            9.32815601193915933109224170038942e-01 2.41325167726604943030377370405404e-02 2.91853549987752510300187402947358e-04
            8.77702204177501554838158881466370e-01 4.39291511332030107372936811316322e-02 6.95803396019414790302226592189072e-04
            8.08938122201322107329701793787535e-01 6.86290872960086950627101032296196e-02 1.30479683517821238962486329882040e-03
            7.29008388828613629328856404754333e-01 9.73397056170658414586682738445234e-02 2.09263503283284519121409950059842e-03
            6.40801775389629346690867350844201e-01 1.29023364563242259883679707854753e-01 2.99427704565782392548412005339742e-03
            5.47506254918818435761806995287770e-01 1.62534949880457996185256774879235e-01 3.91342930989027008165104248860189e-03
            4.52493745081181064637831923391786e-01 1.96663274729912740390602721163305e-01 4.73515280297001368153830824780925e-03
            3.59198224610370542286830186640145e-01 2.30174860047128337914301710043219e-01 5.34172475085933125965587109362787e-03
            2.70991611171386148626538670214359e-01 2.61858518993304756339313144053449e-01 5.62950449682724212568940558298891e-03
            1.91061877798678114714903131243773e-01 2.90569137314361791712968852152699e-01 5.52438777355404066204958013486248e-03
            1.22297795822498389650689887275803e-01 3.15269073477167482977279178157914e-01 4.99361554517962092508698290771463e-03
            6.71843988060842889353807549923658e-02 3.35065707837709947369830842944793e-01 4.05221375096612070748625100691243e-03
            2.77124884633839219105766460415907e-02 3.49243947954787636867024502862478e-01 2.76316966161600438556256165156810e-03
            5.29953250417514176717759255552664e-03 3.57294641943605884026879948578426e-01 1.23296400868474521905771013052799e-03
            9.94700467495824969255124869960127e-01 1.43612885176152226505974507375640e-03 6.08520876952026090533432248208534e-06
            9.72287511536616300134028278989717e-01 7.50985189826078487762561763929625e-03 7.29572095196554903755170595758273e-05
            9.32815601193915933109224170038942e-01 1.82064084780416725328766602842734e-02 2.70361486267154467640072557088615e-04
            8.77702204177501554838158881466370e-01 3.31416767326480551925094175658160e-02 6.44564509513201951314276261228997e-04
            8.08938122201322107329701793787535e-01 5.17761660980942325926967839677673e-02 1.20871173795989318017207736488672e-03
            7.29008388828613629328856404754333e-01 7.34364533252638007621015958648059e-02 1.93853361631251630906036531598602e-03
            6.40801775389629346690867350844201e-01 9.73397056170657998253048504011531e-02 2.77377880924746479315845526514295e-03
            5.47506254918818435761806995287770e-01 1.22622009024523873077505697892775e-01 3.62524480057816074485277368921743e-03
            4.52493745081181064637831923391786e-01 1.48369602146862428204698858280608e-01 4.38645666488133700172680207174380e-03
            3.59198224610370542286830186640145e-01 1.73651905554320390434597243256576e-01 4.94836072041261471748940792281246e-03
            2.70991611171386148626538670214359e-01 1.97555157846122403375588305607380e-01 5.21494839714546111475446110716803e-03
            1.91061877798678114714903131243773e-01 2.19215445073291881339372366710450e-01 5.11757245795654760328607935093714e-03
            1.22297795822498389650689887275803e-01 2.37849934438738086495135348741314e-01 4.62588623882829140521177890832405e-03
            6.71843988060842889353807549923658e-02 2.52785202693344424051957730625873e-01 3.75380917048765035126667299891778e-03
            2.77124884633839219105766460415907e-02 2.63481759273125282216909681665129e-01 2.55969014786410148723505564305469e-03
            5.29953250417514176717759255552664e-03 2.69555482319624617471021110759466e-01 1.14216867300707893180533591959147e-03
            9.94700467495824969255124869960127e-01 1.01253863170281234798442149269704e-03 5.38154146400921456978904203194425e-06
            9.72287511536616300134028278989717e-01 5.29480008428828263083998351135051e-03 6.45207523684330428717301209040613e-05
            9.32815601193915933109224170038942e-01 1.28363773946656902630314434077263e-02 2.39098049668484013583966141247572e-04
            8.77702204177501554838158881466370e-01 2.33664465204858756708361511300609e-02 5.70029848696139949948413772062850e-04
            8.08938122201322107329701793787535e-01 3.65046411479569773983477887213667e-02 1.06894152398630346706420723990050e-03
            7.29008388828613629328856404754333e-01 5.17761660980943158594236308545078e-02 1.71436994697948342017412848292679e-03
            6.40801775389629346690867350844201e-01 6.86290872960087783294369501163601e-02 2.45303098699309763547349305667922e-03
            5.47506254918818435761806995287770e-01 8.64543046273669180123988553532399e-02 3.20603712221253014935662406514894e-03
            4.52493745081181064637831923391786e-01 1.04607573171311307724806738406187e-01 3.87922572851999349768936653504170e-03
            3.59198224610370542286830186640145e-01 1.22432790502669364141041796756326e-01 4.37615362173916170457355079292938e-03
            2.70991611171386148626538670214359e-01 1.39285711700583847427736827739864e-01 4.61191424893696994002922906474851e-03
            1.91061877798678114714903131243773e-01 1.54557236650721102622085822986264e-01 4.52579844351600098228960433743850e-03
            1.22297795822498389650689887275803e-01 1.67695431278192225166279172299255e-01 4.09096869884493369307065790962952e-03
            6.71843988060842889353807549923658e-02 1.78225500404012388022678692323097e-01 3.31973486269558724220951617667197e-03
            2.77124884633839219105766460415907e-02 1.85767077714389772236103226532578e-01 2.26369861536114980876677194032709e-03
            5.29953250417514176717759255552664e-03 1.90049339166975278514470915069978e-01 1.01009321216181686270418893514034e-03
            9.94700467495824969255124869960127e-01 6.48121144150291513333361770321517e-04 4.48338208620572032671834639128150e-06
            9.72287511536616300134028278989717e-01 3.38917625582823515278940718076228e-03 5.37524772951656807469175691771568e-05
            9.32815601193915933109224170038942e-01 8.21650388764377445638409369621513e-03 1.99193468990165024022631623523694e-04
            8.77702204177501554838158881466370e-01 1.49567508630415041404049247830699e-02 4.74893974029307020748996714942791e-04
            8.08938122201322107329701793787535e-01 2.33664465204858548541544394083758e-02 8.90539134910106719607292280471711e-04
            7.29008388828613629328856404754333e-01 3.31416767326480829480850331947295e-02 1.42824793989250681605640558302639e-03
            6.40801775389629346690867350844201e-01 4.39291511332030315539753928533173e-02 2.04362918274331043302916022241789e-03
            5.47506254918818435761806995287770e-01 5.53389876468959804323155537986167e-02 2.67096137743585071086527271688738e-03
            4.52493745081181064637831923391786e-01 6.69588081756024855462072764566983e-02 3.23179729375120414550104364082017e-03
            3.59198224610370542286830186640145e-01 7.83686446892953719745023022369423e-02 3.64578975845572831812613223689823e-03
            2.70991611171386148626538670214359e-01 8.91561190898503275192865658027586e-02 3.84220280844899467198594322780991e-03
            1.91061877798678114714903131243773e-01 9.89313493020125139798537361457420e-02 3.77045941263096529710852067296400e-03
            1.22297795822498389650689887275803e-01 1.07341044959456871632497154678276e-01 3.40820114502388138433275344141293e-03
            6.71843988060842889353807549923658e-02 1.14081291934854583969283225997060e-01 2.76568338530904563737911061593877e-03
            2.77124884633839219105766460415907e-02 1.18908619566670120670792698547302e-01 1.88589568408117609979446349655063e-03
            5.29953250417514176717759255552664e-03 1.21649674678348085343770890176529e-01 8.41512388799933101594941131651240e-04
            9.94700467495824969255124869960127e-01 3.56045905246301838936034966920374e-04 3.42321662715596752889820469922810e-06
            9.72287511536616300134028278989717e-01 1.86184687683297678609251057935126e-03 4.10418676101191064382357753803632e-05
            9.32815601193915933109224170038942e-01 4.51374344293496536362209070603058e-03 1.52091073648618160286949407833390e-04
            8.77702204177501554838158881466370e-01 8.21650388764380047723623334832155e-03 3.62597904165936442064110956096101e-04
            8.08938122201322107329701793787535e-01 1.28363773946657214880540109902540e-02 6.79957299008016611066573808841440e-04
            7.29008388828613629328856404754333e-01 1.82064084780417488607096032637855e-02 1.09051648990260625395054550068608e-03
            6.40801775389629346690867350844201e-01 2.41325167726605845086584878345093e-02 1.56038126209059836366210838320967e-03
            5.47506254918818435761806995287770e-01 3.04005202267927446269091262820439e-02 2.03937099759163168247177999603537e-03
            4.52493745081181064637831923391786e-01 3.67838785792915859418350521536922e-02 2.46758853446941029546879420308869e-03
            3.59198224610370542286830186640145e-01 4.30518820334237148350631230186991e-02 2.78368603886327940777345446576874e-03
            2.70991611171386148626538670214359e-01 4.89779903280425574219059114966512e-02 2.93365416685220691331292819370447e-03
            1.91061877798678114714903131243773e-01 5.43480214114185553042624121644621e-02 2.87887561335605510551283181541748e-03
            1.22297795822498389650689887275803e-01 5.89678949184404832539740937136230e-02 2.60227895013327584564089001162301e-03
            6.71843988060842889353807549923658e-02 6.26706553631493079592473804950714e-02 2.11169451275817739513418480612472e-03
            2.77124884633839219105766460415907e-02 6.53225519292512935010108776623383e-02 1.43994630363791244719029460696902e-03
            5.29953250417514176717759255552664e-03 6.68283529008379811875428799794463e-02 6.42523689908299313645889672130807e-04
            9.94700467495824969255124869960127e-01 1.46863233383278638140290173197400e-04 2.23949801733853346203295849037307e-06
            9.72287511536616300134028278989717e-01 7.67982016833179226104888481785338e-04 2.68499458700907375036884966457151e-05
            9.32815601193915933109224170038942e-01 1.86184687683298914599727691410180e-03 9.94992999242202426878056709114162e-05
            8.77702204177501554838158881466370e-01 3.38917625582826897989718872850062e-03 2.37214694807488313873372609563717e-04
            8.08938122201322107329701793787535e-01 5.29480008428833120309731086194915e-03 4.44833964325664081747302436298241e-04
            7.29008388828613629328856404754333e-01 7.50985189826086640962898854922969e-03 7.13425349023520216466820453860009e-04
            6.40801775389629346690867350844201e-01 9.95427665558288460312397205598245e-03 1.02081495953918778563096481093453e-03
            5.47506254918818435761806995287770e-01 1.25397276903156298238917187859442e-02 1.33417420022249703925876218590929e-03
            4.52493745081181064637831923391786e-01 1.51727607730683076991962110469103e-02 1.61431782806649719470837300860921e-03
            3.59198224610370542286830186640145e-01 1.77582118078010407768996259392225e-02 1.82111155790528234273140117949197e-03
            2.70991611171386148626538670214359e-01 2.02026365651230624398415613995894e-02 1.91922200836024082690189906941214e-03
            1.91061877798678114714903131243773e-01 2.24176883790955863706706452376238e-02 1.88338540340391769924155429549728e-03
            1.22297795822498389650689887275803e-01 2.43233122075556529306794573130901e-02 1.70243346656885337180031836368244e-03
            6.71843988060842889353807549923658e-02 2.58506415865509264762067687115632e-02 1.38148887132381057232799470568807e-03
            2.77124884633839219105766460415907e-02 2.69445064465507339024341604272195e-02 9.42025364824839047360460764224399e-04
            5.29953250417514176717759255552664e-03 2.75656252300006399114096211633296e-02 4.20344572478356019815809441197985e-04
            9.94700467495824969255124869960127e-01 2.80850447628082600061080742470665e-05 9.76778103010141727046539671108771e-07
            9.72287511536616300134028278989717e-01 1.46863233383280264443548901454051e-04 1.17108561784217470600269472846833e-05
            9.32815601193915933109224170038942e-01 3.56045905246308073098526758570870e-04 4.33975545762340611955637514896011e-05
            8.77702204177501554838158881466370e-01 6.48121144150305065860517839126942e-04 1.03463418054530157004500257045265e-04
            8.08938122201322107329701793787535e-01 1.01253863170283273098526422018040e-03 1.94018513284899892391149878179135e-04
            7.29008388828613629328856404754333e-01 1.43612885176155349008231265628410e-03 3.11167169456440630484483911288862e-04
            6.40801775389629346690867350844201e-01 1.90358266676466281359936250083820e-03 4.45238036373905117157323552135040e-04
            5.47506254918818435761806995287770e-01 2.39800530999366217840829129670510e-03 5.81912613580763313746169185236568e-04
            4.52493745081181064637831923391786e-01 2.90152719418148219085451522403218e-03 7.04099889147561481747561451527417e-04
            3.59198224610370542286830186640145e-01 3.39594983741047960409953354599111e-03 7.94294917489837652702089343392799e-04
            2.70991611171386148626538670214359e-01 3.86340365241358936129745238474698e-03 8.37086712320160228339993579993461e-04
            1.91061877798678114714903131243773e-01 4.28699387247230816883059034694270e-03 8.21456240340925794400639414760690e-04
            1.22297795822498389650689887275803e-01 4.65141136002483659289685746784926e-03 7.42532352831608722515399634289679e-04
            6.71843988060842889353807549923658e-02 4.94348659892883228461624156580001e-03 6.02549351959220848978793227956885e-04
            2.77124884633839219105766460415907e-02 5.15266927079185976801056767726550e-03 4.10873214317272826123089712879732e-04
            5.29953250417514176717759255552664e-03 5.27144745941233311414819695528422e-03 1.83337234923727928313785806579972e-04
            ];
        
    case  30
        
        % ALG. DEG.:   30
        % PTS CARD.:  256
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.7e-15
        
        xyw=[
            9.94700467495824969255124869960127e-01 5.27144745941222295920747242803373e-03 9.76778103010145115178328688310128e-07
            9.72287511536616300134028278989717e-01 2.75656252300003658251004168278087e-02 1.17108561784217877176084154910995e-05
            9.32815601193915933109224170038942e-01 6.68283529008377730207257627625950e-02 4.33975545762342102733624682464608e-05
            8.77702204177501554838158881466370e-01 1.21649674678348099221558697990986e-01 1.03463418054530522922733470903012e-04
            8.08938122201322107329701793787535e-01 1.90049339166975139736592836925411e-01 1.94018513284900542912453369481796e-04
            7.29008388828613629328856404754333e-01 2.69555482319624839515626035790774e-01 3.11167169456441714686656396793296e-04
            6.40801775389629346690867350844201e-01 3.57294641943606106071484873609734e-01 4.45238036373906689250473656116469e-04
            5.47506254918818435761806995287770e-01 4.50095739771187941524743791887886e-01 5.81912613580765265310079659144549e-04
            4.52493745081181064637831923391786e-01 5.44604727724637527330742159392685e-01 7.04099889147563975412558168187616e-04
            3.59198224610370542286830186640145e-01 6.37405825552219029717093690123875e-01 7.94294917489840471627737805704328e-04
            2.70991611171386148626538670214359e-01 7.25144985176200296272952527942834e-01 8.37086712320163155685859290855433e-04
            1.91061877798678114714903131243773e-01 8.04651128328849662985078339261236e-01 8.21456240340928613326287877072218e-04
            1.22297795822498389650689887275803e-01 8.73050792817476772889051517267944e-01 7.42532352831611324600613599500321e-04
            6.71843988060842889353807549923658e-02 9.27872114594986974189794182166224e-01 6.02549351959222908962920950415310e-04
            2.77124884633839219105766460415907e-02 9.67134842265824357099290864425711e-01 4.10873214317274289796022568310718e-04
            5.29953250417514176717759255552664e-03 9.89429020036412643079870576912072e-01 1.83337234923728551730034985745021e-04
            9.94700467495824969255124869960127e-01 5.15266927079175308251679510362919e-03 2.23949801733855506137311347503172e-06
            9.72287511536616300134028278989717e-01 2.69445064465504702244658119525411e-02 2.68499458700909916135726729358169e-05
            9.32815601193915933109224170038942e-01 6.53225519292510992119815682599437e-02 9.94992999242211913647065957277960e-05
            8.77702204177501554838158881466370e-01 1.18908619566670148426368314176216e-01 2.37214694807490563592880516985417e-04
            8.08938122201322107329701793787535e-01 1.85767077714389661213800764016923e-01 4.44833964325668310135775129765534e-04
            7.29008388828613629328856404754333e-01 2.63481759273125559772665837954264e-01 7.13425349023526938520289863987500e-04
            6.40801775389629346690867350844201e-01 3.49243947954787914422780659151613e-01 1.02081495953919754345051718047444e-03
            5.47506254918818435761806995287770e-01 4.39954017390866036762986368557904e-01 1.33417420022250961600396301776073e-03
            4.52493745081181064637831923391786e-01 5.32333494145750818482554223010084e-01 1.61431782806651259037922230277218e-03
            3.59198224610370542286830186640145e-01 6.23043563581828552244701313611586e-01 1.82111155790529968996616094756291e-03
            2.70991611171386148626538670214359e-01 7.08805752263490962405967366066761e-01 1.91922200836025904149839682588663e-03
            1.91061877798678114714903131243773e-01 7.86520433822226472386773821199313e-01 1.88338540340393569699761755487089e-03
            1.22297795822498389650689887275803e-01 8.53378891969946096196508733555675e-01 1.70243346656886963483290564624895e-03
            6.71843988060842889353807549923658e-02 9.06964959607365006633017401327379e-01 1.38148887132382358275406453174128e-03
            2.77124884633839219105766460415907e-02 9.45343005090065524598230695119128e-01 9.42025364824847937818275145360758e-04
            5.29953250417514176717759255552664e-03 9.67134842265824468121593326941365e-01 4.20344572478360031363847637564390e-04
            9.94700467495824969255124869960127e-01 4.94348659892872993593115893418144e-03 3.42321662715594550604157608741929e-06
            9.72287511536616300134028278989717e-01 2.58506415865506732065792760977274e-02 4.10418676101188421639562320386574e-05
            9.32815601193915933109224170038942e-01 6.26706553631491136702180710926768e-02 1.52091073648617184504994170879399e-04
            8.77702204177501554838158881466370e-01 1.14081291934854611724858841625974e-01 3.62597904165934111029440112261568e-04
            8.08938122201322107329701793787535e-01 1.78225500404012277000376229807443e-01 6.79957299008012274257883866823704e-04
            7.29008388828613629328856404754333e-01 2.52785202693344701607713886915008e-01 1.09051648990259931505664159345770e-03
            6.40801775389629346690867350844201e-01 3.35065707837710224925586999233929e-01 1.56038126209058838900212151656888e-03
            5.47506254918818435761806995287770e-01 4.22093224854388937572480244853068e-01 2.03937099759161867204571016998216e-03
            4.52493745081181064637831923391786e-01 5.10722376339527550648256237764144e-01 2.46758853446939424927664141762307e-03
            3.59198224610370542286830186640145e-01 5.97749893356205874717090864578495e-01 2.78368603886326162685782570349602e-03
            2.70991611171386148626538670214359e-01 6.80030398500571453546115208155243e-01 2.93365416685218826503556144302820e-03
            1.91061877798678114714903131243773e-01 7.54590100789903517330969862086931e-01 2.87887561335603689091633405894299e-03
            1.22297795822498389650689887275803e-01 8.18734309259061210362062865897315e-01 2.60227895013325893208699923775384e-03
            6.71843988060842889353807549923658e-02 8.70144945830766625149976789543871e-01 2.11169451275816395102724598586974e-03
            2.77124884633839219105766460415907e-02 9.06964959607365006633017401327379e-01 1.43994630363790333989204572873177e-03
            5.29953250417514176717759255552664e-03 9.27872114594987085212096644681878e-01 6.42523689908295193677634227213957e-04
            9.94700467495824969255124869960127e-01 4.65141136002473858102046477824842e-03 4.48338208620572117375129364558184e-06
            9.72287511536616300134028278989717e-01 2.43233122075554065999458686064827e-02 5.37524772951656875231811472115595e-05
            9.32815601193915933109224170038942e-01 5.89678949184402889649447843112284e-02 1.99193468990165051127685935661304e-04
            8.77702204177501554838158881466370e-01 1.07341044959456871632497154678276e-01 4.74893974029307129169213963493235e-04
            8.08938122201322107329701793787535e-01 1.67695431278192086388401094154688e-01 8.90539134910106936447726777572598e-04
            7.29008388828613629328856404754333e-01 2.37849934438738280784164658143709e-01 1.42824793989250724973727457722816e-03
            6.40801775389629346690867350844201e-01 3.15269073477167649510732871931395e-01 2.04362918274331086671002921661966e-03
            5.47506254918818435761806995287770e-01 3.97154757434285576866983547006384e-01 2.67096137743585157822701070529092e-03
            4.52493745081181064637831923391786e-01 4.80547446743216477571536415780429e-01 3.23179729375120457918191263502194e-03
            3.59198224610370542286830186640145e-01 5.62433130700334071860879703308456e-01 3.64578975845572918548787022530178e-03
            2.70991611171386148626538670214359e-01 6.39852269738763523854174763982883e-01 3.84220280844899553934768121621346e-03
            1.91061877798678114714903131243773e-01 7.10006772899309357427455324796028e-01 3.77045941263096616447025866136755e-03
            1.22297795822498389650689887275803e-01 7.70361159218044599938934879901353e-01 3.40820114502388225169449142981648e-03
            6.71843988060842889353807549923658e-02 8.18734309259061099339760403381661e-01 2.76568338530904607105997961014054e-03
            2.77124884633839219105766460415907e-02 8.53378891969945874151903808524366e-01 1.88589568408117653347533249075241e-03
            5.29953250417514176717759255552664e-03 8.73050792817476772889051517267944e-01 8.41512388799933318435375628752126e-04
            9.94700467495824969255124869960127e-01 4.28699387247221969793331552978088e-03 5.38154146400920609945956948894086e-06
            9.72287511536616300134028278989717e-01 2.24176883790953677955126721599299e-02 6.45207523684329344515128723536179e-05
            9.32815601193915933109224170038942e-01 5.43480214114183887708087183909811e-02 2.39098049668483607008151459183409e-04
            8.77702204177501554838158881466370e-01 9.89313493020125417354293517746555e-02 5.70029848696138974166458535108859e-04
            8.08938122201322107329701793787535e-01 1.54557236650721019355358976099524e-01 1.06894152398630173234073126309340e-03
            7.29008388828613629328856404754333e-01 2.19215445073292103383977291741758e-01 1.71436994697948060124848002061526e-03
            6.40801775389629346690867350844201e-01 2.90569137314362013757573777184007e-01 2.45303098699309329866480311466148e-03
            5.47506254918818435761806995287770e-01 3.66039440453814757248096611874644e-01 3.20603712221252494518619613472765e-03
            4.52493745081181064637831923391786e-01 4.42898681747507794170815031975508e-01 3.87922572851998699247633162201510e-03
            3.59198224610370542286830186640145e-01 5.18368984886960260105581710377010e-01 4.37615362173915389831790889729746e-03
            2.70991611171386148626538670214359e-01 5.89722677128030170479178195819259e-01 4.61191424893696213377358716911658e-03
            1.91061877798678114714903131243773e-01 6.54380885550600921440889123914531e-01 4.52579844351599317603396244180658e-03
            1.22297795822498389650689887275803e-01 7.10006772899309468449757787311682e-01 4.09096869884492675417675400240114e-03
            6.71843988060842889353807549923658e-02 7.54590100789903517330969862086931e-01 3.31973486269558160435821925204891e-03
            2.77124884633839219105766460415907e-02 7.86520433822226472386773821199313e-01 2.26369861536114590563895099251113e-03
            5.29953250417514176717759255552664e-03 8.04651128328849774007380801776890e-01 1.01009321216181512798071295833324e-03
            9.94700467495824969255124869960127e-01 3.86340365241350739561321248061176e-03 6.08520876952028885742158187399653e-06
            9.72287511536616300134028278989717e-01 2.02026365651228542730244441827381e-02 7.29572095196558291886959612959629e-05
            9.32815601193915933109224170038942e-01 4.89779903280423770106644099087134e-02 2.70361486267155714472570915418714e-04
            8.77702204177501554838158881466370e-01 8.91561190898502997637109501738450e-02 6.44564509513204878660141972090969e-04
            8.08938122201322107329701793787535e-01 1.39285711700583680894283133966383e-01 1.20871173795989881802337428950977e-03
            7.29008388828613629328856404754333e-01 1.97555157846122514397890768123034e-01 1.93853361631252519951817969712238e-03
            6.40801775389629346690867350844201e-01 2.61858518993304811850464375311276e-01 2.77377880924747736990365609699438e-03
            5.47506254918818435761806995287770e-01 3.29871736056657594016172652118257e-01 3.62524480057817765840666446308660e-03
            4.52493745081181064637831923391786e-01 3.99136652771956423890742371440865e-01 4.38645666488135695104677580502539e-03
            3.59198224610370542286830186640145e-01 4.67149869835308928500694491958711e-01 4.94836072041263726889459562130469e-03
            2.70991611171386148626538670214359e-01 5.31453230982491309219994946033694e-01 5.21494839714548540088312478246735e-03
            1.91061877798678114714903131243773e-01 5.89722677128029837412270808272297e-01 5.11757245795657102205300503783292e-03
            1.22297795822498389650689887275803e-01 6.39852269738763301809569838951575e-01 4.62588623882831222189349063000918e-03
            6.71843988060842889353807549923658e-02 6.80030398500571120479207820608281e-01 3.75380917048766769850143276698873e-03
            2.77124884633839219105766460415907e-02 7.08805752263490518316757516004145e-01 2.55969014786411319661851848650258e-03
            5.29953250417514176717759255552664e-03 7.25144985176200074228347602911526e-01 1.14216867300708413597576385001275e-03
            9.94700467495824969255124869960127e-01 3.39594983741040804675614950269846e-03 6.56894517899701801797894204493211e-06
            9.72287511536616300134028278989717e-01 1.77582118078008603656581243512846e-02 7.87568558284646992552133215603760e-05
            9.32815601193915933109224170038942e-01 4.30518820334235621793972370596748e-02 2.91853549987752781350730524323467e-04
            8.77702204177501554838158881466370e-01 7.83686446892953580967144944224856e-02 6.95803396019415332403312834941289e-04
            8.08938122201322107329701793787535e-01 1.22432790502669239240951526426215e-01 1.30479683517821347382703578432483e-03
            7.29008388828613629328856404754333e-01 1.73651905554320501456899705772230e-01 2.09263503283284692593757547740552e-03
            6.40801775389629346690867350844201e-01 2.30174860047128393425452941301046e-01 2.99427704565782609388846502440629e-03
            5.47506254918818435761806995287770e-01 2.89958795200723484786209382946254e-01 3.91342930989027355109799444221608e-03
            4.52493745081181064637831923391786e-01 3.50842980188906194971565355444909e-01 4.73515280297001715098526020142344e-03
            3.59198224610370542286830186640145e-01 4.10626915342501064287716872058809e-01 5.34172475085933472910282304724205e-03
            2.70991611171386148626538670214359e-01 4.67149869835308984011845723216538e-01 5.62950449682724646249809552500665e-03
            1.91061877798678114714903131243773e-01 5.18368984886960038060976785345702e-01 5.52438777355404499885827007688022e-03
            1.22297795822498389650689887275803e-01 5.62433130700333960838577240792802e-01 4.99361554517962526189567284973236e-03
            6.71843988060842889353807549923658e-02 5.97749893356205652672485939547187e-01 4.05221375096612330957146497212307e-03
            2.77124884633839219105766460415907e-02 6.23043563581828330200096388580278e-01 2.76316966161600655396690662257697e-03
            5.29953250417514176717759255552664e-03 6.37405825552218918694791227608221e-01 1.23296400868474608641944811893154e-03
            9.94700467495824969255124869960127e-01 2.90152719418141843976677307637146e-03 6.81526505898426486447028507797974e-06
            9.72287511536616300134028278989717e-01 1.51727607730681411657425172734293e-02 8.17100513183382121422046462377864e-05
            9.32815601193915933109224170038942e-01 3.67838785792914263472752622874395e-02 3.02797366604869340467320881771229e-04
            8.77702204177501554838158881466370e-01 6.69588081756024161572682373844145e-02 7.21894374758316986588624342147114e-04
            8.08938122201322107329701793787535e-01 1.04607573171311099557989621189336e-01 1.35372362495816880162413653465592e-03
            7.29008388828613629328856404754333e-01 1.48369602146862400449123242651694e-01 2.17110388834903774227247374994931e-03
            6.40801775389629346690867350844201e-01 1.96663274729912629368300258647650e-01 3.10655534033642743638181649146190e-03
            5.47506254918818435761806995287770e-01 2.47743155743588239214858504055883e-01 4.06017363667089854761815814754300e-03
            4.52493745081181064637831923391786e-01 2.99763099175230474102704647521023e-01 4.91270981377870535888874314878194e-03
            3.59198224610370542286830186640145e-01 3.50842980188905917415809199155774e-01 5.54202677252386630146752111159003e-03
            2.70991611171386148626538670214359e-01 3.99136652771956146334986215151730e-01 5.84059757711047691786099278488109e-03
            1.91061877798678114714903131243773e-01 4.42898681747507239059302719397238e-01 5.73153922577435658014533004234181e-03
            1.22297795822498389650689887275803e-01 4.80547446743215977971175334459986e-01 5.18086429642874123080442672062418e-03
            6.71843988060842889353807549923658e-02 5.10722376339526884514441462670220e-01 4.20416216545617307115412231155460e-03
            2.77124884633839219105766460415907e-02 5.32333494145750152348739447916159e-01 2.86678197697065812246663973894556e-03
            5.29953250417514176717759255552664e-03 5.44604727724636972219229846814414e-01 1.27919723766934116844506963417416e-03
            9.94700467495824969255124869960127e-01 2.39800530999360970302314299829050e-03 6.81526505898424961787723450057364e-06
            9.72287511536616300134028278989717e-01 1.25397276903154893112901646645696e-02 8.17100513183380359593516173433159e-05
            9.32815601193915933109224170038942e-01 3.04005202267926093184780000910905e-02 3.02797366604868635735908766193347e-04
            8.77702204177501554838158881466370e-01 5.53389876468959179822704186335613e-02 7.21894374758315360285365613890463e-04
            8.08938122201322107329701793787535e-01 8.64543046273667514789451615797589e-02 1.35372362495816576585805357524350e-03
            7.29008388828613629328856404754333e-01 1.22622009024523845321930082263862e-01 2.17110388834903297178291481372980e-03
            6.40801775389629346690867350844201e-01 1.62534949880457912918529927992495e-01 3.10655534033642049748791258423353e-03
            5.47506254918818435761806995287770e-01 2.04750589337593102978729575625039e-01 4.06017363667088900663904027510398e-03
            4.52493745081181064637831923391786e-01 2.47743155743588211459282888426969e-01 4.91270981377869408318614929953583e-03
            3.59198224610370542286830186640145e-01 2.89958795200723262741604457914946e-01 5.54202677252385415840318927394037e-03
            2.70991611171386148626538670214359e-01 3.29871736056657316460416495829122e-01 5.84059757711046390743492295882788e-03
            1.91061877798678114714903131243773e-01 3.66039440453814257647735530554201e-01 5.73153922577434356971926021628860e-03
            1.22297795822498389650689887275803e-01 3.97154757434285132777773696943768e-01 5.18086429642872995510183287137806e-03
            6.71843988060842889353807549923658e-02 4.22093224854388382460967932274798e-01 4.20416216545616353017500443911558e-03
            2.77124884633839219105766460415907e-02 4.39954017390865426140322824721807e-01 2.86678197697065161725360482591896e-03
            5.29953250417514176717759255552664e-03 4.50095739771187386413231479309616e-01 1.27919723766933856635985566896352e-03
            9.94700467495824969255124869960127e-01 1.90358266676462226443811154297236e-03 6.56894517899701293578125851913008e-06
            9.72287511536616300134028278989717e-01 9.95427665558278051971541344755678e-03 7.87568558284646450451046972851543e-05
            9.32815601193915933109224170038942e-01 2.41325167726604943030377370405404e-02 2.91853549987752510300187402947358e-04
            8.77702204177501554838158881466370e-01 4.39291511332030107372936811316322e-02 6.95803396019414790302226592189072e-04
            8.08938122201322107329701793787535e-01 6.86290872960086950627101032296196e-02 1.30479683517821238962486329882040e-03
            7.29008388828613629328856404754333e-01 9.73397056170658414586682738445234e-02 2.09263503283284519121409950059842e-03
            6.40801775389629346690867350844201e-01 1.29023364563242259883679707854753e-01 2.99427704565782392548412005339742e-03
            5.47506254918818435761806995287770e-01 1.62534949880457996185256774879235e-01 3.91342930989027008165104248860189e-03
            4.52493745081181064637831923391786e-01 1.96663274729912740390602721163305e-01 4.73515280297001368153830824780925e-03
            3.59198224610370542286830186640145e-01 2.30174860047128337914301710043219e-01 5.34172475085933125965587109362787e-03
            2.70991611171386148626538670214359e-01 2.61858518993304756339313144053449e-01 5.62950449682724212568940558298891e-03
            1.91061877798678114714903131243773e-01 2.90569137314361791712968852152699e-01 5.52438777355404066204958013486248e-03
            1.22297795822498389650689887275803e-01 3.15269073477167482977279178157914e-01 4.99361554517962092508698290771463e-03
            6.71843988060842889353807549923658e-02 3.35065707837709947369830842944793e-01 4.05221375096612070748625100691243e-03
            2.77124884633839219105766460415907e-02 3.49243947954787636867024502862478e-01 2.76316966161600438556256165156810e-03
            5.29953250417514176717759255552664e-03 3.57294641943605884026879948578426e-01 1.23296400868474521905771013052799e-03
            9.94700467495824969255124869960127e-01 1.43612885176152226505974507375640e-03 6.08520876952026090533432248208534e-06
            9.72287511536616300134028278989717e-01 7.50985189826078487762561763929625e-03 7.29572095196554903755170595758273e-05
            9.32815601193915933109224170038942e-01 1.82064084780416725328766602842734e-02 2.70361486267154467640072557088615e-04
            8.77702204177501554838158881466370e-01 3.31416767326480551925094175658160e-02 6.44564509513201951314276261228997e-04
            8.08938122201322107329701793787535e-01 5.17761660980942325926967839677673e-02 1.20871173795989318017207736488672e-03
            7.29008388828613629328856404754333e-01 7.34364533252638007621015958648059e-02 1.93853361631251630906036531598602e-03
            6.40801775389629346690867350844201e-01 9.73397056170657998253048504011531e-02 2.77377880924746479315845526514295e-03
            5.47506254918818435761806995287770e-01 1.22622009024523873077505697892775e-01 3.62524480057816074485277368921743e-03
            4.52493745081181064637831923391786e-01 1.48369602146862428204698858280608e-01 4.38645666488133700172680207174380e-03
            3.59198224610370542286830186640145e-01 1.73651905554320390434597243256576e-01 4.94836072041261471748940792281246e-03
            2.70991611171386148626538670214359e-01 1.97555157846122403375588305607380e-01 5.21494839714546111475446110716803e-03
            1.91061877798678114714903131243773e-01 2.19215445073291881339372366710450e-01 5.11757245795654760328607935093714e-03
            1.22297795822498389650689887275803e-01 2.37849934438738086495135348741314e-01 4.62588623882829140521177890832405e-03
            6.71843988060842889353807549923658e-02 2.52785202693344424051957730625873e-01 3.75380917048765035126667299891778e-03
            2.77124884633839219105766460415907e-02 2.63481759273125282216909681665129e-01 2.55969014786410148723505564305469e-03
            5.29953250417514176717759255552664e-03 2.69555482319624617471021110759466e-01 1.14216867300707893180533591959147e-03
            9.94700467495824969255124869960127e-01 1.01253863170281234798442149269704e-03 5.38154146400921456978904203194425e-06
            9.72287511536616300134028278989717e-01 5.29480008428828263083998351135051e-03 6.45207523684330428717301209040613e-05
            9.32815601193915933109224170038942e-01 1.28363773946656902630314434077263e-02 2.39098049668484013583966141247572e-04
            8.77702204177501554838158881466370e-01 2.33664465204858756708361511300609e-02 5.70029848696139949948413772062850e-04
            8.08938122201322107329701793787535e-01 3.65046411479569773983477887213667e-02 1.06894152398630346706420723990050e-03
            7.29008388828613629328856404754333e-01 5.17761660980943158594236308545078e-02 1.71436994697948342017412848292679e-03
            6.40801775389629346690867350844201e-01 6.86290872960087783294369501163601e-02 2.45303098699309763547349305667922e-03
            5.47506254918818435761806995287770e-01 8.64543046273669180123988553532399e-02 3.20603712221253014935662406514894e-03
            4.52493745081181064637831923391786e-01 1.04607573171311307724806738406187e-01 3.87922572851999349768936653504170e-03
            3.59198224610370542286830186640145e-01 1.22432790502669364141041796756326e-01 4.37615362173916170457355079292938e-03
            2.70991611171386148626538670214359e-01 1.39285711700583847427736827739864e-01 4.61191424893696994002922906474851e-03
            1.91061877798678114714903131243773e-01 1.54557236650721102622085822986264e-01 4.52579844351600098228960433743850e-03
            1.22297795822498389650689887275803e-01 1.67695431278192225166279172299255e-01 4.09096869884493369307065790962952e-03
            6.71843988060842889353807549923658e-02 1.78225500404012388022678692323097e-01 3.31973486269558724220951617667197e-03
            2.77124884633839219105766460415907e-02 1.85767077714389772236103226532578e-01 2.26369861536114980876677194032709e-03
            5.29953250417514176717759255552664e-03 1.90049339166975278514470915069978e-01 1.01009321216181686270418893514034e-03
            9.94700467495824969255124869960127e-01 6.48121144150291513333361770321517e-04 4.48338208620572032671834639128150e-06
            9.72287511536616300134028278989717e-01 3.38917625582823515278940718076228e-03 5.37524772951656807469175691771568e-05
            9.32815601193915933109224170038942e-01 8.21650388764377445638409369621513e-03 1.99193468990165024022631623523694e-04
            8.77702204177501554838158881466370e-01 1.49567508630415041404049247830699e-02 4.74893974029307020748996714942791e-04
            8.08938122201322107329701793787535e-01 2.33664465204858548541544394083758e-02 8.90539134910106719607292280471711e-04
            7.29008388828613629328856404754333e-01 3.31416767326480829480850331947295e-02 1.42824793989250681605640558302639e-03
            6.40801775389629346690867350844201e-01 4.39291511332030315539753928533173e-02 2.04362918274331043302916022241789e-03
            5.47506254918818435761806995287770e-01 5.53389876468959804323155537986167e-02 2.67096137743585071086527271688738e-03
            4.52493745081181064637831923391786e-01 6.69588081756024855462072764566983e-02 3.23179729375120414550104364082017e-03
            3.59198224610370542286830186640145e-01 7.83686446892953719745023022369423e-02 3.64578975845572831812613223689823e-03
            2.70991611171386148626538670214359e-01 8.91561190898503275192865658027586e-02 3.84220280844899467198594322780991e-03
            1.91061877798678114714903131243773e-01 9.89313493020125139798537361457420e-02 3.77045941263096529710852067296400e-03
            1.22297795822498389650689887275803e-01 1.07341044959456871632497154678276e-01 3.40820114502388138433275344141293e-03
            6.71843988060842889353807549923658e-02 1.14081291934854583969283225997060e-01 2.76568338530904563737911061593877e-03
            2.77124884633839219105766460415907e-02 1.18908619566670120670792698547302e-01 1.88589568408117609979446349655063e-03
            5.29953250417514176717759255552664e-03 1.21649674678348085343770890176529e-01 8.41512388799933101594941131651240e-04
            9.94700467495824969255124869960127e-01 3.56045905246301838936034966920374e-04 3.42321662715596752889820469922810e-06
            9.72287511536616300134028278989717e-01 1.86184687683297678609251057935126e-03 4.10418676101191064382357753803632e-05
            9.32815601193915933109224170038942e-01 4.51374344293496536362209070603058e-03 1.52091073648618160286949407833390e-04
            8.77702204177501554838158881466370e-01 8.21650388764380047723623334832155e-03 3.62597904165936442064110956096101e-04
            8.08938122201322107329701793787535e-01 1.28363773946657214880540109902540e-02 6.79957299008016611066573808841440e-04
            7.29008388828613629328856404754333e-01 1.82064084780417488607096032637855e-02 1.09051648990260625395054550068608e-03
            6.40801775389629346690867350844201e-01 2.41325167726605845086584878345093e-02 1.56038126209059836366210838320967e-03
            5.47506254918818435761806995287770e-01 3.04005202267927446269091262820439e-02 2.03937099759163168247177999603537e-03
            4.52493745081181064637831923391786e-01 3.67838785792915859418350521536922e-02 2.46758853446941029546879420308869e-03
            3.59198224610370542286830186640145e-01 4.30518820334237148350631230186991e-02 2.78368603886327940777345446576874e-03
            2.70991611171386148626538670214359e-01 4.89779903280425574219059114966512e-02 2.93365416685220691331292819370447e-03
            1.91061877798678114714903131243773e-01 5.43480214114185553042624121644621e-02 2.87887561335605510551283181541748e-03
            1.22297795822498389650689887275803e-01 5.89678949184404832539740937136230e-02 2.60227895013327584564089001162301e-03
            6.71843988060842889353807549923658e-02 6.26706553631493079592473804950714e-02 2.11169451275817739513418480612472e-03
            2.77124884633839219105766460415907e-02 6.53225519292512935010108776623383e-02 1.43994630363791244719029460696902e-03
            5.29953250417514176717759255552664e-03 6.68283529008379811875428799794463e-02 6.42523689908299313645889672130807e-04
            9.94700467495824969255124869960127e-01 1.46863233383278638140290173197400e-04 2.23949801733853346203295849037307e-06
            9.72287511536616300134028278989717e-01 7.67982016833179226104888481785338e-04 2.68499458700907375036884966457151e-05
            9.32815601193915933109224170038942e-01 1.86184687683298914599727691410180e-03 9.94992999242202426878056709114162e-05
            8.77702204177501554838158881466370e-01 3.38917625582826897989718872850062e-03 2.37214694807488313873372609563717e-04
            8.08938122201322107329701793787535e-01 5.29480008428833120309731086194915e-03 4.44833964325664081747302436298241e-04
            7.29008388828613629328856404754333e-01 7.50985189826086640962898854922969e-03 7.13425349023520216466820453860009e-04
            6.40801775389629346690867350844201e-01 9.95427665558288460312397205598245e-03 1.02081495953918778563096481093453e-03
            5.47506254918818435761806995287770e-01 1.25397276903156298238917187859442e-02 1.33417420022249703925876218590929e-03
            4.52493745081181064637831923391786e-01 1.51727607730683076991962110469103e-02 1.61431782806649719470837300860921e-03
            3.59198224610370542286830186640145e-01 1.77582118078010407768996259392225e-02 1.82111155790528234273140117949197e-03
            2.70991611171386148626538670214359e-01 2.02026365651230624398415613995894e-02 1.91922200836024082690189906941214e-03
            1.91061877798678114714903131243773e-01 2.24176883790955863706706452376238e-02 1.88338540340391769924155429549728e-03
            1.22297795822498389650689887275803e-01 2.43233122075556529306794573130901e-02 1.70243346656885337180031836368244e-03
            6.71843988060842889353807549923658e-02 2.58506415865509264762067687115632e-02 1.38148887132381057232799470568807e-03
            2.77124884633839219105766460415907e-02 2.69445064465507339024341604272195e-02 9.42025364824839047360460764224399e-04
            5.29953250417514176717759255552664e-03 2.75656252300006399114096211633296e-02 4.20344572478356019815809441197985e-04
            9.94700467495824969255124869960127e-01 2.80850447628082600061080742470665e-05 9.76778103010141727046539671108771e-07
            9.72287511536616300134028278989717e-01 1.46863233383280264443548901454051e-04 1.17108561784217470600269472846833e-05
            9.32815601193915933109224170038942e-01 3.56045905246308073098526758570870e-04 4.33975545762340611955637514896011e-05
            8.77702204177501554838158881466370e-01 6.48121144150305065860517839126942e-04 1.03463418054530157004500257045265e-04
            8.08938122201322107329701793787535e-01 1.01253863170283273098526422018040e-03 1.94018513284899892391149878179135e-04
            7.29008388828613629328856404754333e-01 1.43612885176155349008231265628410e-03 3.11167169456440630484483911288862e-04
            6.40801775389629346690867350844201e-01 1.90358266676466281359936250083820e-03 4.45238036373905117157323552135040e-04
            5.47506254918818435761806995287770e-01 2.39800530999366217840829129670510e-03 5.81912613580763313746169185236568e-04
            4.52493745081181064637831923391786e-01 2.90152719418148219085451522403218e-03 7.04099889147561481747561451527417e-04
            3.59198224610370542286830186640145e-01 3.39594983741047960409953354599111e-03 7.94294917489837652702089343392799e-04
            2.70991611171386148626538670214359e-01 3.86340365241358936129745238474698e-03 8.37086712320160228339993579993461e-04
            1.91061877798678114714903131243773e-01 4.28699387247230816883059034694270e-03 8.21456240340925794400639414760690e-04
            1.22297795822498389650689887275803e-01 4.65141136002483659289685746784926e-03 7.42532352831608722515399634289679e-04
            6.71843988060842889353807549923658e-02 4.94348659892883228461624156580001e-03 6.02549351959220848978793227956885e-04
            2.77124884633839219105766460415907e-02 5.15266927079185976801056767726550e-03 4.10873214317272826123089712879732e-04
            5.29953250417514176717759255552664e-03 5.27144745941233311414819695528422e-03 1.83337234923727928313785806579972e-04
            ];
        
    case  31
        
        % ALG. DEG.:   31
        % PTS CARD.:  289
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.1e-15
        
        xyw=[
            9.95287737657208571029343602276640e-01 4.69005692640413890392991902444919e-03 6.86977791704559843504471552955160e-07
            9.75337760884383841997191666450817e-01 2.45460241749427786339676771376617e-02 8.25724758900895065177102244424745e-06
            9.40119576863492900642427230195608e-01 5.95982508734905214176080789911794e-02 3.07407928305872675490625911542963e-05
            8.90757001948400684021578399551800e-01 1.08728216385667125765124296776776e-01 7.37883203558057314654441971590870e-05
            8.28835579608345418556325512327021e-01 1.70357848739017270212059429468354e-01 1.39640810387378524852486361140791e-04
            7.56345268543238580249976621416863e-01 2.42506566441074766782293181677232e-01 2.26595470454978367702558728069562e-04
            6.75615881726938205531496350886300e-01 3.22855535207924038143545431012171e-01 3.29007722601925970719316527635101e-04
            5.89242090747923774607386349089211e-01 4.08822310224303875347828807207406e-01 4.37836065125428474439223114345054e-04
            5.00000000000000000000000000000000e-01 4.97643868828604285514671801138320e-01 5.41665964356695720481604805485176e-04
            4.10757909252076280903764882168616e-01 5.86465427432904529148061101295752e-01 6.28086355997652069291237175718834e-04
            3.24384118273061905490806111629354e-01 6.72432202449284366352344477490988e-01 6.85245762906177375770044868374953e-04
            2.43654731456761419750023378583137e-01 7.52781171216133859758201651857235e-01 7.03390453069720005506437754405624e-04
            1.71164420391654581443674487672979e-01 8.24929888918191300817284172808286e-01 6.76187678196028276421392089190476e-04
            1.09242998051599260467270369190373e-01 8.86559521271541584042097383644432e-01 6.01662936675358465087037895813182e-04
            5.98804231365069883352703072887380e-02 9.35689486783718216145189217058942e-01 4.82628873253914411969867082063956e-04
            2.46622391156161580028083335491829e-02 9.70741713482265899948231435701018e-01 3.26556130478533233017779391360591e-04
            4.71226234279131794835393520770594e-03 9.90597680730804563964397857489530e-01 1.45098155066076324893470372856541e-04
            9.95287737657208571029343602276640e-01 4.59604740211799322330277206560822e-03 1.57772847333047117469699560654561e-06
            9.75337760884383841997191666450817e-01 2.40540130774203851637960127618499e-02 1.89637784362634345940862301560514e-05
            9.40119576863492900642427230195608e-01 5.84036378227702837606649666213343e-02 7.05999884235396235349774274325796e-05
            8.90757001948400684021578399551800e-01 1.06548821111943983752645692675287e-01 1.69463897421958509638753587189797e-04
            8.28835579608345418556325512327021e-01 1.66943122527869747262485589089920e-01 3.20702190445563690805730328037271e-04
            7.56345268543238580249976621416863e-01 2.37645660207923520301775965890556e-01 5.20404196440576072464623269553385e-04
            6.75615881726938205531496350886300e-01 3.16384079582903210425826046048314e-01 7.55606452148468109818180149517275e-04
            5.89242090747923774607386349089211e-01 4.00627699475470888668127145137987e-01 1.00554404369514334936785981966523e-03
            5.00000000000000000000000000000000e-01 4.87668880442191920998595833225409e-01 1.24400209922229218646727577635147e-03
            4.10757909252076280903764882168616e-01 5.74710061408912786795610827539349e-01 1.44247709246769968173540465983251e-03
            3.24384118273061905490806111629354e-01 6.58953681301480465037911926629022e-01 1.57375065747552764228167632154509e-03
            2.43654731456761419750023378583137e-01 7.37692100676460293939840084931348e-01 1.61542215640382482572112721186386e-03
            1.71164420391654581443674487672979e-01 8.08394638356514039223554846103070e-01 1.55294765869796666524216277593951e-03
            1.09242998051599260467270369190373e-01 8.68788939772439983144636244105641e-01 1.38179248005236565699505035809125e-03
            5.98804231365069883352703072887380e-02 9.16934123061613703953298681881279e-01 1.10841620293829603086099844233559e-03
            2.46622391156161580028083335491829e-02 9.51283747806963564386251164251007e-01 7.49976071159772481912464403563945e-04
            4.71226234279131794835393520770594e-03 9.70741713482266010970533898216672e-01 3.33235649594215363206856084232754e-04
            9.95287737657208571029343602276640e-01 4.43009007977485016332996892174378e-03 2.41913254542140318379848593077774e-06
            9.75337760884383841997191666450817e-01 2.31854538018793979170517616239522e-02 2.90771792325486597532210181205059e-05
            9.40119576863492900642427230195608e-01 5.62947580614999651404950498090329e-02 1.08251028354221237527850452142530e-04
            8.90757001948400684021578399551800e-01 1.02701481103568928454627950941358e-01 2.59839152590069747284645673701675e-04
            8.28835579608345418556325512327021e-01 1.60915022472687324617268700421846e-01 4.91732968891087844458265276870179e-04
            7.56345268543238580249976621416863e-01 2.29064583037918545471711695427075e-01 7.97936241668864023289520215342918e-04
            6.75615881726938205531496350886300e-01 3.04959860012108063820335246418836e-01 1.15857208057110650130383522338207e-03
            5.89242090747923774607386349089211e-01 3.86161551839394878715694403581438e-01 1.54180162371199218872275249481163e-03
            5.00000000000000000000000000000000e-01 4.70059788431746450321213615097804e-01 1.90742958352558178609925665369929e-03
            4.10757909252076280903764882168616e-01 5.53958025024097855393279132840689e-01 2.21175147650550830979998551129029e-03
            3.24384118273061905490806111629354e-01 6.35159716851384725799789521261118e-01 2.41303335664649724889851967191134e-03
            2.43654731456761419750023378583137e-01 7.11054993825574355170715534768533e-01 2.47692830497124680949605490809518e-03
            1.71164420391654581443674487672979e-01 7.79204554390805603780734145402675e-01 2.38113609914253213442614232064898e-03
            1.09242998051599260467270369190373e-01 8.37418095759924097087889549584361e-01 2.11870370346866569027932669655456e-03
            5.98804231365069883352703072887380e-02 8.83824818801993039585340738995001e-01 1.69953560180110879382353594735378e-03
            2.46622391156161580028083335491829e-02 9.16934123061613592930996219365625e-01 1.14993901212928303391669349053927e-03
            4.71226234279131794835393520770594e-03 9.35689486783718105122886754543288e-01 5.10950533539084148122733086694325e-04
            9.95287737657208571029343602276640e-01 4.19748067685924049208257002874234e-03 3.18290352289908811563826258883569e-06
            9.75337760884383841997191666450817e-01 2.19680621759608743492897531268682e-02 3.82574557108963697648132407103105e-05
            9.40119576863492900642427230195608e-01 5.33389061884767118337791202975495e-02 1.42428152669113109026963193137760e-04
            8.90757001948400684021578399551800e-01 9.73089654282975891064211282355245e-02 3.41875833025915936130151306926450e-04
            8.28835579608345418556325512327021e-01 1.52465905948305946449039538492798e-01 6.46983399884949327181271616638014e-04
            7.56345268543238580249976621416863e-01 2.17037158102967475681310816071345e-01 1.04986148008449453432333609015359e-03
            6.75615881726938205531496350886300e-01 2.88947424672587938054846290469868e-01 1.52435763131777111956122183755724e-03
            5.89242090747923774607386349089211e-01 3.65885483771972608302291973814135e-01 2.02858079397612474370493629294288e-03
            5.00000000000000000000000000000000e-01 4.45378500974200342010789199775900e-01 2.50964518359118740967184102430565e-03
            4.10757909252076280903764882168616e-01 5.24871518176427964696983963222010e-01 2.91004789285753597688488625294667e-03
            3.24384118273061905490806111629354e-01 6.01809577275812634944429646566277e-01 3.17487869206655760889734807506102e-03
            2.43654731456761419750023378583137e-01 6.73719843845433152829116352222627e-01 3.25894661819289355522699658251895e-03
            1.71164420391654581443674487672979e-01 7.38291096000094793083690092316829e-01 3.13291080011605504312965919666567e-03
            1.09242998051599260467270369190373e-01 7.93448036520103205937459733831929e-01 2.78762298267333941820256271171274e-03
            5.98804231365069883352703072887380e-02 8.37418095759924097087889549584361e-01 2.23611470338962517456771195156762e-03
            2.46622391156161580028083335491829e-02 8.68788939772439872122333781589987e-01 1.51299892176342469334027196481429e-03
            4.71226234279131794835393520770594e-03 8.86559521271541584042097383644432e-01 6.72268353508271112629890442491387e-04
            9.95287737657208571029343602276640e-01 3.90569069015411383591351857091922e-03 3.84439786492803795849536319106932e-06
            9.75337760884383841997191666450817e-01 2.04409412518313723938767623167223e-02 4.62084005985160246160990682451342e-05
            9.40119576863492900642427230195608e-01 4.96310252175398355922730786460306e-02 1.72028615409636106677385147811776e-04
            8.90757001948400684021578399551800e-01 9.05444836082506671059988434535626e-02 4.12926974725961548704639136886385e-04
            8.28835579608345418556325512327021e-01 1.41867161583643514033425958587031e-01 7.81444232684786756724937273332898e-04
            7.56345268543238580249976621416863e-01 2.01949710571280610649225195629697e-01 1.26805138876180212119393342362628e-03
            6.75615881726938205531496350886300e-01 2.68861098684595234420413589759846e-01 1.84116080838262873714539225034059e-03
            5.89242090747923774607386349089211e-01 3.40450769793656682349336506376858e-01 2.45017532485325385468999392912792e-03
            5.00000000000000000000000000000000e-01 4.14417789804172709278162756163510e-01 3.03121804230406825461652253750344e-03
            4.10757909252076280903764882168616e-01 4.88384809814688569673535312176682e-01 3.51483537771513435132852976039430e-03
            3.24384118273061905490806111629354e-01 5.59974480923750128624760691309348e-01 3.83470525492681218321378011637535e-03
            2.43654731456761419750023378583137e-01 6.26885869037064780151524701068411e-01 3.93624479371077178740945257118256e-03
            1.71164420391654581443674487672979e-01 6.86968418024701876767323938111076e-01 3.78401528803045880677835555161437e-03
            1.09242998051599260467270369190373e-01 7.38291096000094793083690092316829e-01 3.36696722527504753788574198836159e-03
            5.98804231365069883352703072887380e-02 7.79204554390805714803036607918330e-01 2.70084045262399116346663241472470e-03
            2.46622391156161580028083335491829e-02 8.08394638356514039223554846103070e-01 1.82744144854501272545732604157820e-03
            4.71226234279131794835393520770594e-03 8.24929888918191411839586635323940e-01 8.11984090718487054073870812231917e-04
            9.95287737657208571029343602276640e-01 3.56409732710477383452185584644667e-03 4.38233764675154092386316628804366e-06
            9.75337760884383841997191666450817e-01 1.86531678667783071268182482072007e-02 5.26742602232830301779709092357962e-05
            9.40119576863492900642427230195608e-01 4.52902747176642181403671827411017e-02 1.96100274767555112984118714791748e-04
            8.90757001948400684021578399551800e-01 8.26254246978053719097090379364090e-02 4.70707114684831460452857365694967e-04
            8.28835579608345418556325512327021e-01 1.29459399506173772342876304719539e-01 8.90790339619417794424227619032308e-04
            7.56345268543238580249976621416863e-01 1.84287103295494886223693242754962e-01 1.44548757288690227044980485260339e-03
            6.75615881726938205531496350886300e-01 2.45346393046400601534173802065197e-01 2.09879117817315854338522385091892e-03
            5.89242090747923774607386349089211e-01 3.10674801179520787552945648712921e-01 2.79302401689557940750452402767223e-03
            5.00000000000000000000000000000000e-01 3.78172634271619290124988310708432e-01 3.45537101232113938453771595504804e-03
            4.10757909252076280903764882168616e-01 4.45670467363717681674728510188288e-01 4.00666006461418335721225147949554e-03
            3.24384118273061905490806111629354e-01 5.10998875496837867693500356836012e-01 4.37128876700598658494589798806373e-03
            2.43654731456761419750023378583137e-01 5.72058165247743666270707763032988e-01 4.48703655354656255616063731395116e-03
            1.71164420391654581443674487672979e-01 6.26885869037064780151524701068411e-01 4.31350584285314624793938875768617e-03
            1.09242998051599260467270369190373e-01 6.73719843845433263851418814738281e-01 3.83810098359255343070106647473949e-03
            5.98804231365069883352703072887380e-02 7.11054993825574466193017997284187e-01 3.07876427187255777978558413110477e-03
            2.46622391156161580028083335491829e-02 7.37692100676460293939840084931348e-01 2.08315209262111174226550502908140e-03
            4.71226234279131794835393520770594e-03 7.52781171216133859758201651857235e-01 9.25603585877954326832650622236542e-04
            9.95287737657208571029343602276640e-01 3.18367927765367871723034554065634e-03 4.77942850580479045415277852826286e-06
            9.75337760884383841997191666450817e-01 1.66622004254576329407289136952386e-02 5.74471620232076699095330463151754e-05
            9.40119576863492900642427230195608e-01 4.04561648755533964649799827384413e-02 2.13869244857239703501081140579743e-04
            8.90757001948400684021578399551800e-01 7.38063044511254595647642418043688e-02 5.13358664519479265288259828281525e-04
            8.28835579608345418556325512327021e-01 1.15641400803188035273372236133582e-01 9.71506324947023409445423514085860e-04
            7.56345268543238580249976621416863e-01 1.64617006230100226815693531534635e-01 1.57646559154640342539921427089666e-03
            6.75615881726938205531496350886300e-01 2.19159062085270056607910760249069e-01 2.28896611654924152928769487402860e-03
            5.89242090747923774607386349089211e-01 2.77514567035655135285310279869009e-01 3.04610454049410867005676095686795e-03
            5.00000000000000000000000000000000e-01 3.37807940863469102765748175443150e-01 3.76846789216730634897167817598529e-03
            4.10757909252076280903764882168616e-01 3.98101314691282959223883608501637e-01 4.36971015687974678048943744101962e-03
            3.24384118273061905490806111629354e-01 4.56456819641668065656858743750490e-01 4.76737846879948367478929682761191e-03
            2.43654731456761419750023378583137e-01 5.10998875496837978715802819351666e-01 4.89361435363275427967710839993742e-03
            1.71164420391654581443674487672979e-01 5.59974480923750128624760691309348e-01 4.70435973836241718565309710697875e-03
            1.09242998051599260467270369190373e-01 6.01809577275812856989034571597585e-01 4.18587765886485434208008626910669e-03
            5.98804231365069883352703072887380e-02 6.35159716851384836822091983776772e-01 3.35773619236037247637316482951064e-03
            2.46622391156161580028083335491829e-02 6.58953681301480576060214389144676e-01 2.27190994760078594089103276587593e-03
            4.71226234279131794835393520770594e-03 6.72432202449284588396949402522296e-01 1.00947405700230869417255430420255e-03
            9.95287737657208571029343602276640e-01 2.77666331501913096757583332419017e-03 5.02290608539536931348044521139506e-06
            9.75337760884383841997191666450817e-01 1.45320293390109236270069104079994e-02 6.03736826201306203232115066281693e-05
            9.40119576863492900642427230195608e-01 3.52840657238257873751230420111824e-02 2.24764347906372403528701964248171e-04
            8.90757001948400684021578399551800e-01 6.43705725714957405214633467949170e-02 5.39510604013343020649018377810080e-04
            8.28835579608345418556325512327021e-01 1.00857280933235107789336382211332e-01 1.02099759953513643043199188298331e-03
            7.56345268543238580249976621416863e-01 1.43571623384206009665930992014182e-01 1.65677519887107984816743133649197e-03
            6.75615881726938205531496350886300e-01 1.91140776056640704361200278071919e-01 2.40557251188403551159233906275858e-03
            5.89242090747923774607386349089211e-01 2.42035849238939310135165783322009e-01 3.20128170441624001346658801026024e-03
            5.00000000000000000000000000000000e-01 2.94621045373961887303693174544605e-01 3.96044428433120661303679099773944e-03
            4.10757909252076280903764882168616e-01 3.47206241508984381205493718880462e-01 4.59231552721163331642895144568683e-03
            3.24384118273061905490806111629354e-01 3.98101314691283014735034839759464e-01 5.01024218548982904625210821336623e-03
            2.43654731456761419750023378583137e-01 4.45670467363717737185879741446115e-01 5.14290888263870416691192488656270e-03
            1.71164420391654581443674487672979e-01 4.88384809814688680695837774692336e-01 4.94401310303325784711869772536375e-03
            1.09242998051599260467270369190373e-01 5.24871518176428075719286425737664e-01 4.39911808282032919248383251442647e-03
            5.98804231365069883352703072887380e-02 5.53958025024098077437884057871997e-01 3.52878875231114191857173523203528e-03
            2.46622391156161580028083335491829e-02 5.74710061408912897817913290055003e-01 2.38764745354271305452509288613783e-03
            4.71226234279131794835393520770594e-03 5.86465427432904751192666026327061e-01 1.06089951503769729969361979016185e-03
            9.95287737657208571029343602276640e-01 2.35613117139571448532819886167999e-03 5.10494425241972350266615385261382e-06
            9.75337760884383841997191666450817e-01 1.23311195578081067569797824035049e-02 6.13597544626973877496603493320038e-05
            9.40119576863492900642427230195608e-01 2.99402115682535496787863849021960e-02 2.28435381129206703564385461113773e-04
            8.90757001948400684021578399551800e-01 5.46214990257996579892108002241002e-02 5.48322327802515342362599248815513e-04
            8.28835579608345418556325512327021e-01 8.55822101958272907218372438364895e-02 1.03767335858340366272489774246424e-03
            7.56345268543238580249976621416863e-01 1.21827365728380709875011689291568e-01 1.68383499218117006074324493170025e-03
            6.75615881726938205531496350886300e-01 1.62192059136530897234251824556850e-01 2.44486226888207059185220693109386e-03
            5.89242090747923774607386349089211e-01 2.05378954626038084940731209826481e-01 3.25356762788244326459197175438476e-03
            5.00000000000000000000000000000000e-01 2.50000000000000000000000000000000e-01 4.02512946541259616917152186488238e-03
            4.10757909252076280903764882168616e-01 2.94621045373961831792541943286778e-01 4.66732094077995565001293343243560e-03
            3.24384118273061905490806111629354e-01 3.37807940863469047254596944185323e-01 5.09207351545253646823985249625366e-03
            2.43654731456761419750023378583137e-01 3.78172634271619290124988310708432e-01 5.22690703246111738194956686243131e-03
            1.71164420391654581443674487672979e-01 4.14417789804172709278162756163510e-01 5.02476272602472850969945739052491e-03
            1.09242998051599260467270369190373e-01 4.45378500974200397521940431033727e-01 4.47096803937986186222586226790554e-03
            5.98804231365069883352703072887380e-02 4.70059788431746505832364846355631e-01 3.58642378592196383463308606565079e-03
            2.46622391156161580028083335491829e-02 4.87668880442191920998595833225409e-01 2.42664444398188287876916291452289e-03
            4.71226234279131794835393520770594e-03 4.97643868828604341025823032396147e-01 1.07822698446098197248843586493194e-03
            9.95287737657208571029343602276640e-01 1.93559902777229821992099889627070e-03 5.02290608539535491392034188828930e-06
            9.75337760884383841997191666450817e-01 1.01302097766052898869526543990105e-02 6.03736826201304441403584777336988e-05
            9.40119576863492900642427230195608e-01 2.45963574126813119824497277932096e-02 2.24764347906371725902344160807900e-04
            8.90757001948400684021578399551800e-01 4.48724254801035754569582536532835e-02 5.39510604013341394345759649553429e-04
            8.28835579608345418556325512327021e-01 7.03071394584194875321259132761043e-02 1.02099759953513339466590892357090e-03
            7.56345268543238580249976621416863e-01 1.00083108072555423961880194383411e-01 1.65677519887107486083743790317158e-03
            6.75615881726938205531496350886300e-01 1.33243342216421090107303371041780e-01 2.40557251188402813901756616132843e-03
            5.89242090747923774607386349089211e-01 1.68722060013136887501872251959867e-01 3.20128170441623047248747013782122e-03
            5.00000000000000000000000000000000e-01 2.05378954626038140451882441084308e-01 3.96044428433119533733419714849333e-03
            4.10757909252076280903764882168616e-01 2.42035849238939337890741398950922e-01 4.59231552721161943864114363123008e-03
            3.24384118273061905490806111629354e-01 2.77514567035655135285310279869009e-01 5.01024218548981430110256241050593e-03
            2.43654731456761419750023378583137e-01 3.10674801179520843064096879970748e-01 5.14290888263868942176237908370240e-03
            1.71164420391654581443674487672979e-01 3.40450769793656793371638968892512e-01 4.94401310303324310196915192250344e-03
            1.09242998051599260467270369190373e-01 3.65885483771972774835745667587616e-01 4.39911808282031618205776268837326e-03
            5.98804231365069883352703072887380e-02 3.86161551839394989737996866097092e-01 3.52878875231113151023087937119271e-03
            2.46622391156161580028083335491829e-02 4.00627699475470999690429607653641e-01 2.38764745354270611563118897890945e-03
            4.71226234279131794835393520770594e-03 4.08822310224304041881282500980888e-01 1.06089951503769404708710233364855e-03
            9.95287737657208571029343602276640e-01 1.52858306513775068710692117690542e-03 4.77942850580478706602098951106150e-06
            9.75337760884383841997191666450817e-01 8.00003869015858404267760306538548e-03 5.74471620232076224756880000743564e-05
            9.40119576863492900642427230195608e-01 1.94242582609537098314866909731791e-02 2.13869244857239513765700955616467e-04
            8.90757001948400684021578399551800e-01 3.54366936004738633525512625510601e-02 5.13358664519478831607390834079752e-04
            8.28835579608345418556325512327021e-01 5.55230195884665669869839632610820e-02 9.71506324947022542083685525682313e-04
            7.56345268543238580249976621416863e-01 7.90377252266612345676932704918727e-02 1.57646559154640212435660728829134e-03
            6.75615881726938205531496350886300e-01 1.05225056187791779493956312308001e-01 2.28896611654923979456421889722151e-03
            5.89242090747923774607386349089211e-01 1.33243342216421117862878986670694e-01 3.04610454049410606797154699165731e-03
            5.00000000000000000000000000000000e-01 1.62192059136530952745403055814677e-01 3.76846789216730331320559521657287e-03
            4.10757909252076280903764882168616e-01 1.91140776056640759872351509329746e-01 4.36971015687974331104248548740543e-03
            3.24384118273061905490806111629354e-01 2.19159062085270084363486375877983e-01 4.76737846879947933798060688559417e-03
            2.43654731456761419750023378583137e-01 2.45346393046400684800900648951938e-01 4.89361435363274994286841845791969e-03
            1.71164420391654581443674487672979e-01 2.68861098684595345442716052275500e-01 4.70435973836241284884440716496101e-03
            1.09242998051599260467270369190373e-01 2.88947424672588104588299984243349e-01 4.18587765886485173999487230389605e-03
            5.98804231365069883352703072887380e-02 3.04959860012108230353788940192317e-01 3.35773619236036987428795086430000e-03
            2.46622391156161580028083335491829e-02 3.16384079582903321448128508563968e-01 2.27190994760078420616755678906884e-03
            4.71226234279131794835393520770594e-03 3.22855535207924204676999124785652e-01 1.00947405700230782681081631579900e-03
            9.95287737657208571029343602276640e-01 1.14816501568665513613454187691332e-03 4.38233764675154770012674432244637e-06
            9.75337760884383841997191666450817e-01 6.00907124883790551977957861140567e-03 5.26742602232831182693974236830314e-05
            9.40119576863492900642427230195608e-01 1.45901484188428812172055870632903e-02 1.96100274767555411139716148305467e-04
            8.90757001948400684021578399551800e-01 2.66175733537939440687125625117915e-02 4.70707114684832219394378105548071e-04
            8.28835579608345418556325512327021e-01 4.17050208854808160396920868606685e-02 8.90790339619419203887051850188072e-04
            7.56345268543238580249976621416863e-01 5.93676281612665196485423280137184e-02 1.44548757288690443885414982361226e-03
            6.75615881726938205531496350886300e-01 7.90377252266612068121176548629592e-02 2.09879117817316201283217580453311e-03
            5.89242090747923774607386349089211e-01 1.00083108072555396206304578754498e-01 2.79302401689558417799408296389174e-03
            5.00000000000000000000000000000000e-01 1.21827365728380709875011689291568e-01 3.45537101232114502238901287967110e-03
            4.10757909252076280903764882168616e-01 1.43571623384205981910355376385269e-01 4.00666006461419029610615538672391e-03
            3.24384118273061905490806111629354e-01 1.64617006230100199060117915905721e-01 4.37128876700599439120153988369566e-03
            2.43654731456761419750023378583137e-01 1.84287103295494886223693242754962e-01 4.48703655354656949505454122117953e-03
            1.71164420391654581443674487672979e-01 2.01949710571280610649225195629697e-01 4.31350584285315318683329266491455e-03
            1.09242998051599260467270369190373e-01 2.17037158102967503436886431700259e-01 3.83810098359255950223323239356432e-03
            5.98804231365069883352703072887380e-02 2.29064583037918573227287311055989e-01 3.07876427187256255027514306732428e-03
            2.46622391156161580028083335491829e-02 2.37645660207923520301775965890556e-01 2.08315209262111521171245698269558e-03
            4.71226234279131794835393520770594e-03 2.42506566441074794537868797306146e-01 9.25603585877955844715692101942750e-04
            9.95287737657208571029343602276640e-01 8.06571652637315243163096400991208e-04 3.84439786492803457036357417386796e-06
            9.75337760884383841997191666450817e-01 4.22129786378484112008280249028758e-03 4.62084005985159771822540220043152e-05
            9.40119576863492900642427230195608e-01 1.02493979189672602958527392047472e-02 1.72028615409635944047059274986111e-04
            8.90757001948400684021578399551800e-01 1.86985144433486488724227569946379e-02 4.12926974725961169233878766959833e-04
            8.28835579608345418556325512327021e-01 2.92972588080110570019076732251051e-02 7.81444232684785997783416533479794e-04
            7.56345268543238580249976621416863e-01 4.17050208854808160396920868606685e-02 1.26805138876180082015132644102096e-03
            6.75615881726938205531496350886300e-01 5.55230195884665531091961554466252e-02 1.84116080838262678558148177643261e-03
            5.89242090747923774607386349089211e-01 7.03071394584194736543381054616475e-02 2.45017532485325125260477996391728e-03
            5.00000000000000000000000000000000e-01 8.55822101958272907218372438364895e-02 3.03121804230406521885043957809103e-03
            4.10757909252076280903764882168616e-01 1.00857280933235080033760766582418e-01 3.51483537771513088188157780678011e-03
            3.24384118273061905490806111629354e-01 1.15641400803188007517796620504669e-01 3.83470525492680828008595916855938e-03
            2.43654731456761419750023378583137e-01 1.29459399506173772342876304719539e-01 3.93624479371076831796250061756837e-03
            1.71164420391654581443674487672979e-01 1.41867161583643514033425958587031e-01 3.78401528803045490365053460379841e-03
            1.09242998051599260467270369190373e-01 1.52465905948305946449039538492798e-01 3.36696722527504406843879003474740e-03
            5.98804231365069883352703072887380e-02 1.60915022472687352372844316050760e-01 2.70084045262398856138141844951406e-03
            2.46622391156161580028083335491829e-02 1.66943122527869747262485589089920e-01 1.82744144854501099073385006477110e-03
            4.71226234279131794835393520770594e-03 1.70357848739017297967635045097268e-01 8.11984090718486186712132823828370e-04
            9.95287737657208571029343602276640e-01 5.14781665932188695414262191718535e-04 3.18290352289908938618768347028620e-06
            9.75337760884383841997191666450817e-01 2.69417693965533699626546670913285e-03 3.82574557108963833173403967791160e-05
            9.40119576863492900642427230195608e-01 6.54151694803038405434669755322830e-03 1.42428152669113163237071817412982e-04
            8.90757001948400684021578399551800e-01 1.19340326233017251372769962358689e-02 3.41875833025916044550368555476894e-04
            8.28835579608345418556325512327021e-01 1.86985144433486384640819011337953e-02 6.46983399884949544021706113738901e-04
            7.56345268543238580249976621416863e-01 2.66175733537939301909247546973347e-02 1.04986148008449496800420508435536e-03
            6.75615881726938205531496350886300e-01 3.54366936004738355969756469221466e-02 1.52435763131777177008252532885990e-03
            5.89242090747923774607386349089211e-01 4.48724254801035407624887341171416e-02 2.02858079397612561106667428134642e-03
            5.00000000000000000000000000000000e-01 5.46214990257996302336351845951867e-02 2.50964518359118871071444800691097e-03
            4.10757909252076280903764882168616e-01 6.43705725714956988880999233515467e-02 2.91004789285753684424662424135022e-03
            3.24384118273061905490806111629354e-01 7.38063044511254179314008183609985e-02 3.17487869206655847625908606346457e-03
            2.43654731456761419750023378583137e-01 8.26254246978053302763456144930387e-02 3.25894661819289442258873457092250e-03
            1.71164420391654581443674487672979e-01 9.05444836082506254726354200101923e-02 3.13291080011605634417226617927099e-03
            1.09242998051599260467270369190373e-01 9.73089654282975474730577047921543e-02 2.78762298267334028556430070011629e-03
            5.98804231365069883352703072887380e-02 1.02701481103568886821264527497988e-01 2.23611470338962604192944993997116e-03
            2.46622391156161580028083335491829e-02 1.06548821111943928241494461417460e-01 1.51299892176342512702114095901607e-03
            4.71226234279131794835393520770594e-03 1.08728216385667084131760873333405e-01 6.72268353508271329470324939592274e-04
            9.95287737657208571029343602276640e-01 2.82172263016578482065777055964872e-04 2.41913254542140784247969582942961e-06
            9.75337760884383841997191666450817e-01 1.47678531373681277798215472074617e-03 2.90771792325487139633296423957276e-05
            9.40119576863492900642427230195608e-01 3.58566507500712814554555407653424e-03 1.08251028354221427263230637105806e-04
            8.90757001948400684021578399551800e-01 6.54151694803037538072931766919282e-03 2.59839152590070235175623292178670e-04
            8.28835579608345418556325512327021e-01 1.02493979189672412138945034598692e-02 4.91732968891088711820003265273726e-04
            7.56345268543238580249976621416863e-01 1.45901484188428534616299714343768e-02 7.97936241668865432752344446498682e-04
            6.75615881726938205531496350886300e-01 1.94242582609536681981232675298088e-02 1.15857208057110866970818019439093e-03
            5.89242090747923774607386349089211e-01 2.45963574126812599407454484889968e-02 1.54180162371199500764840095712316e-03
            5.00000000000000000000000000000000e-01 2.99402115682534941676351536443690e-02 1.90742958352558525554620860731347e-03
            4.10757909252076280903764882168616e-01 3.52840657238257179861840029388986e-02 2.21175147650551221292780645910625e-03
            3.24384118273061905490806111629354e-01 4.04561648755533131982531358517008e-02 2.41303335664650158570720961392908e-03
            2.43654731456761419750023378583137e-01 4.52902747176641348736403358543612e-02 2.47692830497125157998561384431468e-03
            1.71164420391654581443674487672979e-01 4.96310252175397453866523278520617e-02 2.38113609914253647123483226266671e-03
            1.09242998051599260467270369190373e-01 5.33389061884766216281583695035806e-02 2.11870370346866915972627865016875e-03
            5.98804231365069883352703072887380e-02 5.62947580614998679959803951078356e-02 1.69953560180111182958961890676619e-03
            2.46622391156161580028083335491829e-02 5.84036378227701796772564080129087e-02 1.14993901212928520232103846154814e-03
            4.71226234279131794835393520770594e-03 5.95982508734904173341995203827537e-02 5.10950533539085123904688323648315e-04
            9.95287737657208571029343602276640e-01 1.16214940673435815116261438095790e-04 1.57772847333046863359815384364460e-06
            9.75337760884383841997191666450817e-01 6.08226038195829000684855536462692e-04 1.89637784362634041009001290012392e-05
            9.40119576863492900642427230195608e-01 1.47678531373681212746085122944351e-03 7.05999884235395151147601788821362e-05
            8.90757001948400684021578399551800e-01 2.69417693965533222577590777291334e-03 1.69463897421958238588210465813688e-04
            8.28835579608345418556325512327021e-01 4.22129786378483157910368461784856e-03 3.20702190445563148704644085285054e-04
            7.56345268543238580249976621416863e-01 6.00907124883789250935350878535246e-03 5.20404196440575205102885281149838e-04
            6.75615881726938205531496350886300e-01 8.00003869015856322599589134370035e-03 7.55606452148466917195790415462398e-04
            5.89242090747923774607386349089211e-01 1.01302097766052656008239907237112e-02 1.00554404369514183148481833995902e-03
            5.00000000000000000000000000000000e-01 1.23311195578080790014041667745914e-02 1.24400209922229023490336530244349e-03
            4.10757909252076280903764882168616e-01 1.45320293390108889325373908718575e-02 1.44247709246769729649062519172276e-03
            3.24384118273061905490806111629354e-01 1.66622004254575913073654902518683e-02 1.57375065747552504019646235633445e-03
            2.43654731456761419750023378583137e-01 1.86531678667782654934548247638304e-02 1.61542215640382222363591324665322e-03
            1.71164420391654581443674487672979e-01 2.04409412518313272910663869197379e-02 1.55294765869796406315694881072886e-03
            1.09242998051599260467270369190373e-01 2.19680621759608292464793777298837e-02 1.38179248005236348859070538708238e-03
            5.98804231365069883352703072887380e-02 2.31854538018793493447944342733535e-02 1.10841620293829429613752246552849e-03
            2.46622391156161580028083335491829e-02 2.40540130774203296526447815040228e-02 7.49976071159771289290074669509067e-04
            4.71226234279131794835393520770594e-03 2.45460241749427265922633978334488e-02 3.33235649594214821105769841480537e-04
            9.95287737657208571029343602276640e-01 2.22054163872896432100050717606265e-05 6.86977791704566196251575960207703e-07
            9.75337760884383841997191666450817e-01 1.16214940673433335003791877504398e-04 8.25724758900902688473627533127797e-06
            9.40119576863492900642427230195608e-01 2.82172263016572356323502512864820e-04 3.07407928305875453758692905648076e-05
            8.90757001948400684021578399551800e-01 5.14781665932176769190364851169761e-04 7.37883203558064226443291566681637e-05
            8.28835579608345418556325512327021e-01 8.06571652637296269625077904663613e-04 1.39640810387379825895093343746112e-04
            7.56345268543238580249976621416863e-01 1.14816501568662803108022973930247e-03 2.26595470454980454791740762665597e-04
            6.75615881726938205531496350886300e-01 1.52858306513771404107349116685555e-03 3.29007722601929006485399487047516e-04
            5.89242090747923774607386349089211e-01 1.93559902777225203290845101378181e-03 4.37836065125432540197369934986682e-04
            5.00000000000000000000000000000000e-01 2.35613117139565897417696760385297e-03 5.41665964356700707811598238805573e-04
            4.10757909252076280903764882168616e-01 2.77666331501906504808374620552058e-03 6.28086355997657923982968597442778e-04
            3.24384118273061905490806111629354e-01 3.18367927765360325675914054954774e-03 6.85245762906183664142645284300670e-04
            2.43654731456761419750023378583137e-01 3.56409732710469013411413996550436e-03 7.03390453069726402299255418881785e-04
            1.71164420391654581443674487672979e-01 3.90569069015402189556929180014322e-03 6.76187678196034456373775256565750e-04
            1.09242998051599260467270369190373e-01 4.19748067685914161284443935073796e-03 6.01662936675363994518117571885796e-04
            5.98804231365069883352703072887380e-02 4.43009007977474607992141031331812e-03 4.82628873253918857198774272632136e-04
            2.46622391156161580028083335491829e-02 4.59604740211788480308552351516482e-03 3.26556130478536214573753726497785e-04
            4.71226234279131794835393520770594e-03 4.69005692640402874898919449719870e-03 1.45098155066077653041131667599473e-04
            ];
        
    case  32
        
        % ALG. DEG.:   32
        % PTS CARD.:  289
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.1e-15
        
        xyw=[
            9.95287737657208571029343602276640e-01 4.69005692640413890392991902444919e-03 6.86977791704559843504471552955160e-07
            9.75337760884383841997191666450817e-01 2.45460241749427786339676771376617e-02 8.25724758900895065177102244424745e-06
            9.40119576863492900642427230195608e-01 5.95982508734905214176080789911794e-02 3.07407928305872675490625911542963e-05
            8.90757001948400684021578399551800e-01 1.08728216385667125765124296776776e-01 7.37883203558057314654441971590870e-05
            8.28835579608345418556325512327021e-01 1.70357848739017270212059429468354e-01 1.39640810387378524852486361140791e-04
            7.56345268543238580249976621416863e-01 2.42506566441074766782293181677232e-01 2.26595470454978367702558728069562e-04
            6.75615881726938205531496350886300e-01 3.22855535207924038143545431012171e-01 3.29007722601925970719316527635101e-04
            5.89242090747923774607386349089211e-01 4.08822310224303875347828807207406e-01 4.37836065125428474439223114345054e-04
            5.00000000000000000000000000000000e-01 4.97643868828604285514671801138320e-01 5.41665964356695720481604805485176e-04
            4.10757909252076280903764882168616e-01 5.86465427432904529148061101295752e-01 6.28086355997652069291237175718834e-04
            3.24384118273061905490806111629354e-01 6.72432202449284366352344477490988e-01 6.85245762906177375770044868374953e-04
            2.43654731456761419750023378583137e-01 7.52781171216133859758201651857235e-01 7.03390453069720005506437754405624e-04
            1.71164420391654581443674487672979e-01 8.24929888918191300817284172808286e-01 6.76187678196028276421392089190476e-04
            1.09242998051599260467270369190373e-01 8.86559521271541584042097383644432e-01 6.01662936675358465087037895813182e-04
            5.98804231365069883352703072887380e-02 9.35689486783718216145189217058942e-01 4.82628873253914411969867082063956e-04
            2.46622391156161580028083335491829e-02 9.70741713482265899948231435701018e-01 3.26556130478533233017779391360591e-04
            4.71226234279131794835393520770594e-03 9.90597680730804563964397857489530e-01 1.45098155066076324893470372856541e-04
            9.95287737657208571029343602276640e-01 4.59604740211799322330277206560822e-03 1.57772847333047117469699560654561e-06
            9.75337760884383841997191666450817e-01 2.40540130774203851637960127618499e-02 1.89637784362634345940862301560514e-05
            9.40119576863492900642427230195608e-01 5.84036378227702837606649666213343e-02 7.05999884235396235349774274325796e-05
            8.90757001948400684021578399551800e-01 1.06548821111943983752645692675287e-01 1.69463897421958509638753587189797e-04
            8.28835579608345418556325512327021e-01 1.66943122527869747262485589089920e-01 3.20702190445563690805730328037271e-04
            7.56345268543238580249976621416863e-01 2.37645660207923520301775965890556e-01 5.20404196440576072464623269553385e-04
            6.75615881726938205531496350886300e-01 3.16384079582903210425826046048314e-01 7.55606452148468109818180149517275e-04
            5.89242090747923774607386349089211e-01 4.00627699475470888668127145137987e-01 1.00554404369514334936785981966523e-03
            5.00000000000000000000000000000000e-01 4.87668880442191920998595833225409e-01 1.24400209922229218646727577635147e-03
            4.10757909252076280903764882168616e-01 5.74710061408912786795610827539349e-01 1.44247709246769968173540465983251e-03
            3.24384118273061905490806111629354e-01 6.58953681301480465037911926629022e-01 1.57375065747552764228167632154509e-03
            2.43654731456761419750023378583137e-01 7.37692100676460293939840084931348e-01 1.61542215640382482572112721186386e-03
            1.71164420391654581443674487672979e-01 8.08394638356514039223554846103070e-01 1.55294765869796666524216277593951e-03
            1.09242998051599260467270369190373e-01 8.68788939772439983144636244105641e-01 1.38179248005236565699505035809125e-03
            5.98804231365069883352703072887380e-02 9.16934123061613703953298681881279e-01 1.10841620293829603086099844233559e-03
            2.46622391156161580028083335491829e-02 9.51283747806963564386251164251007e-01 7.49976071159772481912464403563945e-04
            4.71226234279131794835393520770594e-03 9.70741713482266010970533898216672e-01 3.33235649594215363206856084232754e-04
            9.95287737657208571029343602276640e-01 4.43009007977485016332996892174378e-03 2.41913254542140318379848593077774e-06
            9.75337760884383841997191666450817e-01 2.31854538018793979170517616239522e-02 2.90771792325486597532210181205059e-05
            9.40119576863492900642427230195608e-01 5.62947580614999651404950498090329e-02 1.08251028354221237527850452142530e-04
            8.90757001948400684021578399551800e-01 1.02701481103568928454627950941358e-01 2.59839152590069747284645673701675e-04
            8.28835579608345418556325512327021e-01 1.60915022472687324617268700421846e-01 4.91732968891087844458265276870179e-04
            7.56345268543238580249976621416863e-01 2.29064583037918545471711695427075e-01 7.97936241668864023289520215342918e-04
            6.75615881726938205531496350886300e-01 3.04959860012108063820335246418836e-01 1.15857208057110650130383522338207e-03
            5.89242090747923774607386349089211e-01 3.86161551839394878715694403581438e-01 1.54180162371199218872275249481163e-03
            5.00000000000000000000000000000000e-01 4.70059788431746450321213615097804e-01 1.90742958352558178609925665369929e-03
            4.10757909252076280903764882168616e-01 5.53958025024097855393279132840689e-01 2.21175147650550830979998551129029e-03
            3.24384118273061905490806111629354e-01 6.35159716851384725799789521261118e-01 2.41303335664649724889851967191134e-03
            2.43654731456761419750023378583137e-01 7.11054993825574355170715534768533e-01 2.47692830497124680949605490809518e-03
            1.71164420391654581443674487672979e-01 7.79204554390805603780734145402675e-01 2.38113609914253213442614232064898e-03
            1.09242998051599260467270369190373e-01 8.37418095759924097087889549584361e-01 2.11870370346866569027932669655456e-03
            5.98804231365069883352703072887380e-02 8.83824818801993039585340738995001e-01 1.69953560180110879382353594735378e-03
            2.46622391156161580028083335491829e-02 9.16934123061613592930996219365625e-01 1.14993901212928303391669349053927e-03
            4.71226234279131794835393520770594e-03 9.35689486783718105122886754543288e-01 5.10950533539084148122733086694325e-04
            9.95287737657208571029343602276640e-01 4.19748067685924049208257002874234e-03 3.18290352289908811563826258883569e-06
            9.75337760884383841997191666450817e-01 2.19680621759608743492897531268682e-02 3.82574557108963697648132407103105e-05
            9.40119576863492900642427230195608e-01 5.33389061884767118337791202975495e-02 1.42428152669113109026963193137760e-04
            8.90757001948400684021578399551800e-01 9.73089654282975891064211282355245e-02 3.41875833025915936130151306926450e-04
            8.28835579608345418556325512327021e-01 1.52465905948305946449039538492798e-01 6.46983399884949327181271616638014e-04
            7.56345268543238580249976621416863e-01 2.17037158102967475681310816071345e-01 1.04986148008449453432333609015359e-03
            6.75615881726938205531496350886300e-01 2.88947424672587938054846290469868e-01 1.52435763131777111956122183755724e-03
            5.89242090747923774607386349089211e-01 3.65885483771972608302291973814135e-01 2.02858079397612474370493629294288e-03
            5.00000000000000000000000000000000e-01 4.45378500974200342010789199775900e-01 2.50964518359118740967184102430565e-03
            4.10757909252076280903764882168616e-01 5.24871518176427964696983963222010e-01 2.91004789285753597688488625294667e-03
            3.24384118273061905490806111629354e-01 6.01809577275812634944429646566277e-01 3.17487869206655760889734807506102e-03
            2.43654731456761419750023378583137e-01 6.73719843845433152829116352222627e-01 3.25894661819289355522699658251895e-03
            1.71164420391654581443674487672979e-01 7.38291096000094793083690092316829e-01 3.13291080011605504312965919666567e-03
            1.09242998051599260467270369190373e-01 7.93448036520103205937459733831929e-01 2.78762298267333941820256271171274e-03
            5.98804231365069883352703072887380e-02 8.37418095759924097087889549584361e-01 2.23611470338962517456771195156762e-03
            2.46622391156161580028083335491829e-02 8.68788939772439872122333781589987e-01 1.51299892176342469334027196481429e-03
            4.71226234279131794835393520770594e-03 8.86559521271541584042097383644432e-01 6.72268353508271112629890442491387e-04
            9.95287737657208571029343602276640e-01 3.90569069015411383591351857091922e-03 3.84439786492803795849536319106932e-06
            9.75337760884383841997191666450817e-01 2.04409412518313723938767623167223e-02 4.62084005985160246160990682451342e-05
            9.40119576863492900642427230195608e-01 4.96310252175398355922730786460306e-02 1.72028615409636106677385147811776e-04
            8.90757001948400684021578399551800e-01 9.05444836082506671059988434535626e-02 4.12926974725961548704639136886385e-04
            8.28835579608345418556325512327021e-01 1.41867161583643514033425958587031e-01 7.81444232684786756724937273332898e-04
            7.56345268543238580249976621416863e-01 2.01949710571280610649225195629697e-01 1.26805138876180212119393342362628e-03
            6.75615881726938205531496350886300e-01 2.68861098684595234420413589759846e-01 1.84116080838262873714539225034059e-03
            5.89242090747923774607386349089211e-01 3.40450769793656682349336506376858e-01 2.45017532485325385468999392912792e-03
            5.00000000000000000000000000000000e-01 4.14417789804172709278162756163510e-01 3.03121804230406825461652253750344e-03
            4.10757909252076280903764882168616e-01 4.88384809814688569673535312176682e-01 3.51483537771513435132852976039430e-03
            3.24384118273061905490806111629354e-01 5.59974480923750128624760691309348e-01 3.83470525492681218321378011637535e-03
            2.43654731456761419750023378583137e-01 6.26885869037064780151524701068411e-01 3.93624479371077178740945257118256e-03
            1.71164420391654581443674487672979e-01 6.86968418024701876767323938111076e-01 3.78401528803045880677835555161437e-03
            1.09242998051599260467270369190373e-01 7.38291096000094793083690092316829e-01 3.36696722527504753788574198836159e-03
            5.98804231365069883352703072887380e-02 7.79204554390805714803036607918330e-01 2.70084045262399116346663241472470e-03
            2.46622391156161580028083335491829e-02 8.08394638356514039223554846103070e-01 1.82744144854501272545732604157820e-03
            4.71226234279131794835393520770594e-03 8.24929888918191411839586635323940e-01 8.11984090718487054073870812231917e-04
            9.95287737657208571029343602276640e-01 3.56409732710477383452185584644667e-03 4.38233764675154092386316628804366e-06
            9.75337760884383841997191666450817e-01 1.86531678667783071268182482072007e-02 5.26742602232830301779709092357962e-05
            9.40119576863492900642427230195608e-01 4.52902747176642181403671827411017e-02 1.96100274767555112984118714791748e-04
            8.90757001948400684021578399551800e-01 8.26254246978053719097090379364090e-02 4.70707114684831460452857365694967e-04
            8.28835579608345418556325512327021e-01 1.29459399506173772342876304719539e-01 8.90790339619417794424227619032308e-04
            7.56345268543238580249976621416863e-01 1.84287103295494886223693242754962e-01 1.44548757288690227044980485260339e-03
            6.75615881726938205531496350886300e-01 2.45346393046400601534173802065197e-01 2.09879117817315854338522385091892e-03
            5.89242090747923774607386349089211e-01 3.10674801179520787552945648712921e-01 2.79302401689557940750452402767223e-03
            5.00000000000000000000000000000000e-01 3.78172634271619290124988310708432e-01 3.45537101232113938453771595504804e-03
            4.10757909252076280903764882168616e-01 4.45670467363717681674728510188288e-01 4.00666006461418335721225147949554e-03
            3.24384118273061905490806111629354e-01 5.10998875496837867693500356836012e-01 4.37128876700598658494589798806373e-03
            2.43654731456761419750023378583137e-01 5.72058165247743666270707763032988e-01 4.48703655354656255616063731395116e-03
            1.71164420391654581443674487672979e-01 6.26885869037064780151524701068411e-01 4.31350584285314624793938875768617e-03
            1.09242998051599260467270369190373e-01 6.73719843845433263851418814738281e-01 3.83810098359255343070106647473949e-03
            5.98804231365069883352703072887380e-02 7.11054993825574466193017997284187e-01 3.07876427187255777978558413110477e-03
            2.46622391156161580028083335491829e-02 7.37692100676460293939840084931348e-01 2.08315209262111174226550502908140e-03
            4.71226234279131794835393520770594e-03 7.52781171216133859758201651857235e-01 9.25603585877954326832650622236542e-04
            9.95287737657208571029343602276640e-01 3.18367927765367871723034554065634e-03 4.77942850580479045415277852826286e-06
            9.75337760884383841997191666450817e-01 1.66622004254576329407289136952386e-02 5.74471620232076699095330463151754e-05
            9.40119576863492900642427230195608e-01 4.04561648755533964649799827384413e-02 2.13869244857239703501081140579743e-04
            8.90757001948400684021578399551800e-01 7.38063044511254595647642418043688e-02 5.13358664519479265288259828281525e-04
            8.28835579608345418556325512327021e-01 1.15641400803188035273372236133582e-01 9.71506324947023409445423514085860e-04
            7.56345268543238580249976621416863e-01 1.64617006230100226815693531534635e-01 1.57646559154640342539921427089666e-03
            6.75615881726938205531496350886300e-01 2.19159062085270056607910760249069e-01 2.28896611654924152928769487402860e-03
            5.89242090747923774607386349089211e-01 2.77514567035655135285310279869009e-01 3.04610454049410867005676095686795e-03
            5.00000000000000000000000000000000e-01 3.37807940863469102765748175443150e-01 3.76846789216730634897167817598529e-03
            4.10757909252076280903764882168616e-01 3.98101314691282959223883608501637e-01 4.36971015687974678048943744101962e-03
            3.24384118273061905490806111629354e-01 4.56456819641668065656858743750490e-01 4.76737846879948367478929682761191e-03
            2.43654731456761419750023378583137e-01 5.10998875496837978715802819351666e-01 4.89361435363275427967710839993742e-03
            1.71164420391654581443674487672979e-01 5.59974480923750128624760691309348e-01 4.70435973836241718565309710697875e-03
            1.09242998051599260467270369190373e-01 6.01809577275812856989034571597585e-01 4.18587765886485434208008626910669e-03
            5.98804231365069883352703072887380e-02 6.35159716851384836822091983776772e-01 3.35773619236037247637316482951064e-03
            2.46622391156161580028083335491829e-02 6.58953681301480576060214389144676e-01 2.27190994760078594089103276587593e-03
            4.71226234279131794835393520770594e-03 6.72432202449284588396949402522296e-01 1.00947405700230869417255430420255e-03
            9.95287737657208571029343602276640e-01 2.77666331501913096757583332419017e-03 5.02290608539536931348044521139506e-06
            9.75337760884383841997191666450817e-01 1.45320293390109236270069104079994e-02 6.03736826201306203232115066281693e-05
            9.40119576863492900642427230195608e-01 3.52840657238257873751230420111824e-02 2.24764347906372403528701964248171e-04
            8.90757001948400684021578399551800e-01 6.43705725714957405214633467949170e-02 5.39510604013343020649018377810080e-04
            8.28835579608345418556325512327021e-01 1.00857280933235107789336382211332e-01 1.02099759953513643043199188298331e-03
            7.56345268543238580249976621416863e-01 1.43571623384206009665930992014182e-01 1.65677519887107984816743133649197e-03
            6.75615881726938205531496350886300e-01 1.91140776056640704361200278071919e-01 2.40557251188403551159233906275858e-03
            5.89242090747923774607386349089211e-01 2.42035849238939310135165783322009e-01 3.20128170441624001346658801026024e-03
            5.00000000000000000000000000000000e-01 2.94621045373961887303693174544605e-01 3.96044428433120661303679099773944e-03
            4.10757909252076280903764882168616e-01 3.47206241508984381205493718880462e-01 4.59231552721163331642895144568683e-03
            3.24384118273061905490806111629354e-01 3.98101314691283014735034839759464e-01 5.01024218548982904625210821336623e-03
            2.43654731456761419750023378583137e-01 4.45670467363717737185879741446115e-01 5.14290888263870416691192488656270e-03
            1.71164420391654581443674487672979e-01 4.88384809814688680695837774692336e-01 4.94401310303325784711869772536375e-03
            1.09242998051599260467270369190373e-01 5.24871518176428075719286425737664e-01 4.39911808282032919248383251442647e-03
            5.98804231365069883352703072887380e-02 5.53958025024098077437884057871997e-01 3.52878875231114191857173523203528e-03
            2.46622391156161580028083335491829e-02 5.74710061408912897817913290055003e-01 2.38764745354271305452509288613783e-03
            4.71226234279131794835393520770594e-03 5.86465427432904751192666026327061e-01 1.06089951503769729969361979016185e-03
            9.95287737657208571029343602276640e-01 2.35613117139571448532819886167999e-03 5.10494425241972350266615385261382e-06
            9.75337760884383841997191666450817e-01 1.23311195578081067569797824035049e-02 6.13597544626973877496603493320038e-05
            9.40119576863492900642427230195608e-01 2.99402115682535496787863849021960e-02 2.28435381129206703564385461113773e-04
            8.90757001948400684021578399551800e-01 5.46214990257996579892108002241002e-02 5.48322327802515342362599248815513e-04
            8.28835579608345418556325512327021e-01 8.55822101958272907218372438364895e-02 1.03767335858340366272489774246424e-03
            7.56345268543238580249976621416863e-01 1.21827365728380709875011689291568e-01 1.68383499218117006074324493170025e-03
            6.75615881726938205531496350886300e-01 1.62192059136530897234251824556850e-01 2.44486226888207059185220693109386e-03
            5.89242090747923774607386349089211e-01 2.05378954626038084940731209826481e-01 3.25356762788244326459197175438476e-03
            5.00000000000000000000000000000000e-01 2.50000000000000000000000000000000e-01 4.02512946541259616917152186488238e-03
            4.10757909252076280903764882168616e-01 2.94621045373961831792541943286778e-01 4.66732094077995565001293343243560e-03
            3.24384118273061905490806111629354e-01 3.37807940863469047254596944185323e-01 5.09207351545253646823985249625366e-03
            2.43654731456761419750023378583137e-01 3.78172634271619290124988310708432e-01 5.22690703246111738194956686243131e-03
            1.71164420391654581443674487672979e-01 4.14417789804172709278162756163510e-01 5.02476272602472850969945739052491e-03
            1.09242998051599260467270369190373e-01 4.45378500974200397521940431033727e-01 4.47096803937986186222586226790554e-03
            5.98804231365069883352703072887380e-02 4.70059788431746505832364846355631e-01 3.58642378592196383463308606565079e-03
            2.46622391156161580028083335491829e-02 4.87668880442191920998595833225409e-01 2.42664444398188287876916291452289e-03
            4.71226234279131794835393520770594e-03 4.97643868828604341025823032396147e-01 1.07822698446098197248843586493194e-03
            9.95287737657208571029343602276640e-01 1.93559902777229821992099889627070e-03 5.02290608539535491392034188828930e-06
            9.75337760884383841997191666450817e-01 1.01302097766052898869526543990105e-02 6.03736826201304441403584777336988e-05
            9.40119576863492900642427230195608e-01 2.45963574126813119824497277932096e-02 2.24764347906371725902344160807900e-04
            8.90757001948400684021578399551800e-01 4.48724254801035754569582536532835e-02 5.39510604013341394345759649553429e-04
            8.28835579608345418556325512327021e-01 7.03071394584194875321259132761043e-02 1.02099759953513339466590892357090e-03
            7.56345268543238580249976621416863e-01 1.00083108072555423961880194383411e-01 1.65677519887107486083743790317158e-03
            6.75615881726938205531496350886300e-01 1.33243342216421090107303371041780e-01 2.40557251188402813901756616132843e-03
            5.89242090747923774607386349089211e-01 1.68722060013136887501872251959867e-01 3.20128170441623047248747013782122e-03
            5.00000000000000000000000000000000e-01 2.05378954626038140451882441084308e-01 3.96044428433119533733419714849333e-03
            4.10757909252076280903764882168616e-01 2.42035849238939337890741398950922e-01 4.59231552721161943864114363123008e-03
            3.24384118273061905490806111629354e-01 2.77514567035655135285310279869009e-01 5.01024218548981430110256241050593e-03
            2.43654731456761419750023378583137e-01 3.10674801179520843064096879970748e-01 5.14290888263868942176237908370240e-03
            1.71164420391654581443674487672979e-01 3.40450769793656793371638968892512e-01 4.94401310303324310196915192250344e-03
            1.09242998051599260467270369190373e-01 3.65885483771972774835745667587616e-01 4.39911808282031618205776268837326e-03
            5.98804231365069883352703072887380e-02 3.86161551839394989737996866097092e-01 3.52878875231113151023087937119271e-03
            2.46622391156161580028083335491829e-02 4.00627699475470999690429607653641e-01 2.38764745354270611563118897890945e-03
            4.71226234279131794835393520770594e-03 4.08822310224304041881282500980888e-01 1.06089951503769404708710233364855e-03
            9.95287737657208571029343602276640e-01 1.52858306513775068710692117690542e-03 4.77942850580478706602098951106150e-06
            9.75337760884383841997191666450817e-01 8.00003869015858404267760306538548e-03 5.74471620232076224756880000743564e-05
            9.40119576863492900642427230195608e-01 1.94242582609537098314866909731791e-02 2.13869244857239513765700955616467e-04
            8.90757001948400684021578399551800e-01 3.54366936004738633525512625510601e-02 5.13358664519478831607390834079752e-04
            8.28835579608345418556325512327021e-01 5.55230195884665669869839632610820e-02 9.71506324947022542083685525682313e-04
            7.56345268543238580249976621416863e-01 7.90377252266612345676932704918727e-02 1.57646559154640212435660728829134e-03
            6.75615881726938205531496350886300e-01 1.05225056187791779493956312308001e-01 2.28896611654923979456421889722151e-03
            5.89242090747923774607386349089211e-01 1.33243342216421117862878986670694e-01 3.04610454049410606797154699165731e-03
            5.00000000000000000000000000000000e-01 1.62192059136530952745403055814677e-01 3.76846789216730331320559521657287e-03
            4.10757909252076280903764882168616e-01 1.91140776056640759872351509329746e-01 4.36971015687974331104248548740543e-03
            3.24384118273061905490806111629354e-01 2.19159062085270084363486375877983e-01 4.76737846879947933798060688559417e-03
            2.43654731456761419750023378583137e-01 2.45346393046400684800900648951938e-01 4.89361435363274994286841845791969e-03
            1.71164420391654581443674487672979e-01 2.68861098684595345442716052275500e-01 4.70435973836241284884440716496101e-03
            1.09242998051599260467270369190373e-01 2.88947424672588104588299984243349e-01 4.18587765886485173999487230389605e-03
            5.98804231365069883352703072887380e-02 3.04959860012108230353788940192317e-01 3.35773619236036987428795086430000e-03
            2.46622391156161580028083335491829e-02 3.16384079582903321448128508563968e-01 2.27190994760078420616755678906884e-03
            4.71226234279131794835393520770594e-03 3.22855535207924204676999124785652e-01 1.00947405700230782681081631579900e-03
            9.95287737657208571029343602276640e-01 1.14816501568665513613454187691332e-03 4.38233764675154770012674432244637e-06
            9.75337760884383841997191666450817e-01 6.00907124883790551977957861140567e-03 5.26742602232831182693974236830314e-05
            9.40119576863492900642427230195608e-01 1.45901484188428812172055870632903e-02 1.96100274767555411139716148305467e-04
            8.90757001948400684021578399551800e-01 2.66175733537939440687125625117915e-02 4.70707114684832219394378105548071e-04
            8.28835579608345418556325512327021e-01 4.17050208854808160396920868606685e-02 8.90790339619419203887051850188072e-04
            7.56345268543238580249976621416863e-01 5.93676281612665196485423280137184e-02 1.44548757288690443885414982361226e-03
            6.75615881726938205531496350886300e-01 7.90377252266612068121176548629592e-02 2.09879117817316201283217580453311e-03
            5.89242090747923774607386349089211e-01 1.00083108072555396206304578754498e-01 2.79302401689558417799408296389174e-03
            5.00000000000000000000000000000000e-01 1.21827365728380709875011689291568e-01 3.45537101232114502238901287967110e-03
            4.10757909252076280903764882168616e-01 1.43571623384205981910355376385269e-01 4.00666006461419029610615538672391e-03
            3.24384118273061905490806111629354e-01 1.64617006230100199060117915905721e-01 4.37128876700599439120153988369566e-03
            2.43654731456761419750023378583137e-01 1.84287103295494886223693242754962e-01 4.48703655354656949505454122117953e-03
            1.71164420391654581443674487672979e-01 2.01949710571280610649225195629697e-01 4.31350584285315318683329266491455e-03
            1.09242998051599260467270369190373e-01 2.17037158102967503436886431700259e-01 3.83810098359255950223323239356432e-03
            5.98804231365069883352703072887380e-02 2.29064583037918573227287311055989e-01 3.07876427187256255027514306732428e-03
            2.46622391156161580028083335491829e-02 2.37645660207923520301775965890556e-01 2.08315209262111521171245698269558e-03
            4.71226234279131794835393520770594e-03 2.42506566441074794537868797306146e-01 9.25603585877955844715692101942750e-04
            9.95287737657208571029343602276640e-01 8.06571652637315243163096400991208e-04 3.84439786492803457036357417386796e-06
            9.75337760884383841997191666450817e-01 4.22129786378484112008280249028758e-03 4.62084005985159771822540220043152e-05
            9.40119576863492900642427230195608e-01 1.02493979189672602958527392047472e-02 1.72028615409635944047059274986111e-04
            8.90757001948400684021578399551800e-01 1.86985144433486488724227569946379e-02 4.12926974725961169233878766959833e-04
            8.28835579608345418556325512327021e-01 2.92972588080110570019076732251051e-02 7.81444232684785997783416533479794e-04
            7.56345268543238580249976621416863e-01 4.17050208854808160396920868606685e-02 1.26805138876180082015132644102096e-03
            6.75615881726938205531496350886300e-01 5.55230195884665531091961554466252e-02 1.84116080838262678558148177643261e-03
            5.89242090747923774607386349089211e-01 7.03071394584194736543381054616475e-02 2.45017532485325125260477996391728e-03
            5.00000000000000000000000000000000e-01 8.55822101958272907218372438364895e-02 3.03121804230406521885043957809103e-03
            4.10757909252076280903764882168616e-01 1.00857280933235080033760766582418e-01 3.51483537771513088188157780678011e-03
            3.24384118273061905490806111629354e-01 1.15641400803188007517796620504669e-01 3.83470525492680828008595916855938e-03
            2.43654731456761419750023378583137e-01 1.29459399506173772342876304719539e-01 3.93624479371076831796250061756837e-03
            1.71164420391654581443674487672979e-01 1.41867161583643514033425958587031e-01 3.78401528803045490365053460379841e-03
            1.09242998051599260467270369190373e-01 1.52465905948305946449039538492798e-01 3.36696722527504406843879003474740e-03
            5.98804231365069883352703072887380e-02 1.60915022472687352372844316050760e-01 2.70084045262398856138141844951406e-03
            2.46622391156161580028083335491829e-02 1.66943122527869747262485589089920e-01 1.82744144854501099073385006477110e-03
            4.71226234279131794835393520770594e-03 1.70357848739017297967635045097268e-01 8.11984090718486186712132823828370e-04
            9.95287737657208571029343602276640e-01 5.14781665932188695414262191718535e-04 3.18290352289908938618768347028620e-06
            9.75337760884383841997191666450817e-01 2.69417693965533699626546670913285e-03 3.82574557108963833173403967791160e-05
            9.40119576863492900642427230195608e-01 6.54151694803038405434669755322830e-03 1.42428152669113163237071817412982e-04
            8.90757001948400684021578399551800e-01 1.19340326233017251372769962358689e-02 3.41875833025916044550368555476894e-04
            8.28835579608345418556325512327021e-01 1.86985144433486384640819011337953e-02 6.46983399884949544021706113738901e-04
            7.56345268543238580249976621416863e-01 2.66175733537939301909247546973347e-02 1.04986148008449496800420508435536e-03
            6.75615881726938205531496350886300e-01 3.54366936004738355969756469221466e-02 1.52435763131777177008252532885990e-03
            5.89242090747923774607386349089211e-01 4.48724254801035407624887341171416e-02 2.02858079397612561106667428134642e-03
            5.00000000000000000000000000000000e-01 5.46214990257996302336351845951867e-02 2.50964518359118871071444800691097e-03
            4.10757909252076280903764882168616e-01 6.43705725714956988880999233515467e-02 2.91004789285753684424662424135022e-03
            3.24384118273061905490806111629354e-01 7.38063044511254179314008183609985e-02 3.17487869206655847625908606346457e-03
            2.43654731456761419750023378583137e-01 8.26254246978053302763456144930387e-02 3.25894661819289442258873457092250e-03
            1.71164420391654581443674487672979e-01 9.05444836082506254726354200101923e-02 3.13291080011605634417226617927099e-03
            1.09242998051599260467270369190373e-01 9.73089654282975474730577047921543e-02 2.78762298267334028556430070011629e-03
            5.98804231365069883352703072887380e-02 1.02701481103568886821264527497988e-01 2.23611470338962604192944993997116e-03
            2.46622391156161580028083335491829e-02 1.06548821111943928241494461417460e-01 1.51299892176342512702114095901607e-03
            4.71226234279131794835393520770594e-03 1.08728216385667084131760873333405e-01 6.72268353508271329470324939592274e-04
            9.95287737657208571029343602276640e-01 2.82172263016578482065777055964872e-04 2.41913254542140784247969582942961e-06
            9.75337760884383841997191666450817e-01 1.47678531373681277798215472074617e-03 2.90771792325487139633296423957276e-05
            9.40119576863492900642427230195608e-01 3.58566507500712814554555407653424e-03 1.08251028354221427263230637105806e-04
            8.90757001948400684021578399551800e-01 6.54151694803037538072931766919282e-03 2.59839152590070235175623292178670e-04
            8.28835579608345418556325512327021e-01 1.02493979189672412138945034598692e-02 4.91732968891088711820003265273726e-04
            7.56345268543238580249976621416863e-01 1.45901484188428534616299714343768e-02 7.97936241668865432752344446498682e-04
            6.75615881726938205531496350886300e-01 1.94242582609536681981232675298088e-02 1.15857208057110866970818019439093e-03
            5.89242090747923774607386349089211e-01 2.45963574126812599407454484889968e-02 1.54180162371199500764840095712316e-03
            5.00000000000000000000000000000000e-01 2.99402115682534941676351536443690e-02 1.90742958352558525554620860731347e-03
            4.10757909252076280903764882168616e-01 3.52840657238257179861840029388986e-02 2.21175147650551221292780645910625e-03
            3.24384118273061905490806111629354e-01 4.04561648755533131982531358517008e-02 2.41303335664650158570720961392908e-03
            2.43654731456761419750023378583137e-01 4.52902747176641348736403358543612e-02 2.47692830497125157998561384431468e-03
            1.71164420391654581443674487672979e-01 4.96310252175397453866523278520617e-02 2.38113609914253647123483226266671e-03
            1.09242998051599260467270369190373e-01 5.33389061884766216281583695035806e-02 2.11870370346866915972627865016875e-03
            5.98804231365069883352703072887380e-02 5.62947580614998679959803951078356e-02 1.69953560180111182958961890676619e-03
            2.46622391156161580028083335491829e-02 5.84036378227701796772564080129087e-02 1.14993901212928520232103846154814e-03
            4.71226234279131794835393520770594e-03 5.95982508734904173341995203827537e-02 5.10950533539085123904688323648315e-04
            9.95287737657208571029343602276640e-01 1.16214940673435815116261438095790e-04 1.57772847333046863359815384364460e-06
            9.75337760884383841997191666450817e-01 6.08226038195829000684855536462692e-04 1.89637784362634041009001290012392e-05
            9.40119576863492900642427230195608e-01 1.47678531373681212746085122944351e-03 7.05999884235395151147601788821362e-05
            8.90757001948400684021578399551800e-01 2.69417693965533222577590777291334e-03 1.69463897421958238588210465813688e-04
            8.28835579608345418556325512327021e-01 4.22129786378483157910368461784856e-03 3.20702190445563148704644085285054e-04
            7.56345268543238580249976621416863e-01 6.00907124883789250935350878535246e-03 5.20404196440575205102885281149838e-04
            6.75615881726938205531496350886300e-01 8.00003869015856322599589134370035e-03 7.55606452148466917195790415462398e-04
            5.89242090747923774607386349089211e-01 1.01302097766052656008239907237112e-02 1.00554404369514183148481833995902e-03
            5.00000000000000000000000000000000e-01 1.23311195578080790014041667745914e-02 1.24400209922229023490336530244349e-03
            4.10757909252076280903764882168616e-01 1.45320293390108889325373908718575e-02 1.44247709246769729649062519172276e-03
            3.24384118273061905490806111629354e-01 1.66622004254575913073654902518683e-02 1.57375065747552504019646235633445e-03
            2.43654731456761419750023378583137e-01 1.86531678667782654934548247638304e-02 1.61542215640382222363591324665322e-03
            1.71164420391654581443674487672979e-01 2.04409412518313272910663869197379e-02 1.55294765869796406315694881072886e-03
            1.09242998051599260467270369190373e-01 2.19680621759608292464793777298837e-02 1.38179248005236348859070538708238e-03
            5.98804231365069883352703072887380e-02 2.31854538018793493447944342733535e-02 1.10841620293829429613752246552849e-03
            2.46622391156161580028083335491829e-02 2.40540130774203296526447815040228e-02 7.49976071159771289290074669509067e-04
            4.71226234279131794835393520770594e-03 2.45460241749427265922633978334488e-02 3.33235649594214821105769841480537e-04
            9.95287737657208571029343602276640e-01 2.22054163872896432100050717606265e-05 6.86977791704566196251575960207703e-07
            9.75337760884383841997191666450817e-01 1.16214940673433335003791877504398e-04 8.25724758900902688473627533127797e-06
            9.40119576863492900642427230195608e-01 2.82172263016572356323502512864820e-04 3.07407928305875453758692905648076e-05
            8.90757001948400684021578399551800e-01 5.14781665932176769190364851169761e-04 7.37883203558064226443291566681637e-05
            8.28835579608345418556325512327021e-01 8.06571652637296269625077904663613e-04 1.39640810387379825895093343746112e-04
            7.56345268543238580249976621416863e-01 1.14816501568662803108022973930247e-03 2.26595470454980454791740762665597e-04
            6.75615881726938205531496350886300e-01 1.52858306513771404107349116685555e-03 3.29007722601929006485399487047516e-04
            5.89242090747923774607386349089211e-01 1.93559902777225203290845101378181e-03 4.37836065125432540197369934986682e-04
            5.00000000000000000000000000000000e-01 2.35613117139565897417696760385297e-03 5.41665964356700707811598238805573e-04
            4.10757909252076280903764882168616e-01 2.77666331501906504808374620552058e-03 6.28086355997657923982968597442778e-04
            3.24384118273061905490806111629354e-01 3.18367927765360325675914054954774e-03 6.85245762906183664142645284300670e-04
            2.43654731456761419750023378583137e-01 3.56409732710469013411413996550436e-03 7.03390453069726402299255418881785e-04
            1.71164420391654581443674487672979e-01 3.90569069015402189556929180014322e-03 6.76187678196034456373775256565750e-04
            1.09242998051599260467270369190373e-01 4.19748067685914161284443935073796e-03 6.01662936675363994518117571885796e-04
            5.98804231365069883352703072887380e-02 4.43009007977474607992141031331812e-03 4.82628873253918857198774272632136e-04
            2.46622391156161580028083335491829e-02 4.59604740211788480308552351516482e-03 3.26556130478536214573753726497785e-04
            4.71226234279131794835393520770594e-03 4.69005692640402874898919449719870e-03 1.45098155066077653041131667599473e-04
            ];
        
    case  33
        
        % ALG. DEG.:   33
        % PTS CARD.:  324
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.2e-15
        
        xyw=[
            9.95782584210465393503852737921989e-01 4.19962919359279096553905219479930e-03 4.92649033616424885908191208527818e-07
            9.77911974785698800971545097127091e-01 2.19948708280027112327292826421399e-02 5.93411562953544289222233137914841e-06
            9.46301233248777906581494789861608e-01 5.34722967244469563063979933303926e-02 2.21778540467582421270853476613638e-05
            9.01852479486261682239955916884355e-01 9.77335916110200625395165729969449e-02 5.35386036087621701071680158090516e-05
            8.45843521530176500888842383574229e-01 1.53506336503465762310938202972466e-01 1.02096101662044511744140140585557e-04
            7.79885415536973658490182970126625e-01 2.19186269739005068490911298795254e-01 1.67294741203757619130848555144553e-04
            7.05875580731421203850572965166066e-01 2.92883974298667748570323965395801e-01 2.45863457315829004971863280815114e-04
            6.25943112845752769324292330566095e-01 3.72479333732178730098638652634691e-01 3.32069253567048754734875037897268e-04
            5.42387506520867757053849800286116e-01 4.55682551323644990937111742823618e-01 4.18278873565718957178855008649521e-04
            4.57612493479132464990755124745192e-01 5.40100032886820069499833607551409e-01 4.95767135942527131277357543837070e-04
            3.74056887154247230675707669433905e-01 6.23303250478286607894062854029471e-01 5.55681420116221035099268732437849e-04
            2.94124419268578907171729497349588e-01 7.02898609911797533911226310010534e-01 5.90053050151441156427611911539088e-04
            2.20114584463026341509817029873375e-01 7.76596314471460269501790207868908e-01 5.92740045277459861629287818374223e-04
            1.54156478469823443600006385167944e-01 8.42276247706999603437338919320609e-01 5.60192649842030819182236989206558e-04
            9.81475205137383732711953143734718e-02 8.98048992599445372597699588368414e-01 4.91952544089335579158284161849224e-04
            5.36987667512220934185052101383917e-02 9.42310287486018416380773032869911e-01 3.90827050693508856304275722592934e-04
            2.20880252143010324950012090994278e-02 9.73787713382462838396236293192487e-01 2.62723474714638013070844291263484e-04
            4.21741578953443996269356830453034e-03 9.91582955016872658049464916985016e-01 1.16320361160663073387712407313899e-04
            9.95782584210465393503852737921989e-01 4.12426140323617421157287310506945e-03 1.13304076349582783674843806942523e-06
            9.77911974785698800971545097127091e-01 2.16001443564335406100074266078082e-02 1.36478394247624472629745739915563e-05
            9.46301233248777906581494789861608e-01 5.25126670372442205803054093848914e-02 5.10067227725507051560790983124605e-05
            9.01852479486261682239955916884355e-01 9.59796356059097743873564922978403e-02 1.23133135701232682493455605765575e-04
            8.45843521530176500888842383574229e-01 1.50751466286434099295377109228866e-01 2.34810254529349104850310947512071e-04
            7.79885415536973658490182970126625e-01 2.15252687971371536868048224278027e-01 3.84760241811266698146176690187303e-04
            7.05875580731421203850572965166066e-01 2.87627791679632649479714245899231e-01 5.65459993594034340987086739716005e-04
            6.25943112845752769324292330566095e-01 3.65794709199201206750728943006834e-01 7.63724223374899003138893682773869e-04
            5.42387506520867757053849800286116e-01 4.47504737184785894577032649976900e-01 9.61997247371189096593058209805349e-04
            4.57612493479132464990755124745192e-01 5.30407237600912684349907522118883e-01 1.14021206963701841365088629487445e-03
            3.74056887154247230675707669433905e-01 6.12117265586497594220816154120257e-01 1.27800859749405571613722898405285e-03
            2.94124419268578907171729497349588e-01 6.90284183106065984958377157454379e-01 1.35705971762275947982423485171921e-03
            2.20114584463026341509817029873375e-01 7.62659286814327264103496872849064e-01 1.36323952272000099503190817529230e-03
            1.54156478469823443600006385167944e-01 8.27160508499264590653865525382571e-01 1.28838394956836072360872869069226e-03
            9.81475205137383732711953143734718e-02 8.81932339179789082095339836087078e-01 1.13143891111879761439407143797098e-03
            5.36987667512220934185052101383917e-02 9.25399307748454580391239687742200e-01 8.98860953125052973035347125829730e-04
            2.20880252143010324950012090994278e-02 9.56311830429265374853287084988551e-01 6.04236253532814825313590478828019e-04
            4.21741578953443996269356830453034e-03 9.73787713382462727373933830676833e-01 2.67524549580673500111249740740504e-04
            9.95782584210465393503852737921989e-01 3.99094576275946678195483130480170e-03 1.74181340640683344320898719159718e-06
            9.77911974785698800971545097127091e-01 2.09019255003232741485508228151957e-02 2.09807011754758379726195499648256e-05
            9.46301233248777906581494789861608e-01 5.08152092006199374019104197941488e-02 7.84121775707257633123564710686537e-05
            9.01852479486261682239955916884355e-01 9.28771197024603478720194971174351e-02 1.89291465450535604381210319502316e-04
            8.45843521530176500888842383574229e-01 1.45878465689282615924682318109262e-01 3.60971698881447312836495688870286e-04
            7.79885415536973658490182970126625e-01 2.08294702733404057948263243815745e-01 5.91488469815911583303547072176798e-04
            7.05875580731421203850572965166066e-01 2.78330300682436693193722021533176e-01 8.69276578002339339948378871270052e-04
            6.25943112845752769324292330566095e-01 3.53970493619263082418768817660748e-01 1.17406640072481799688675074833100e-03
            5.42387506520867757053849800286116e-01 4.33039266929351129853387192270020e-01 1.47886974271581729360736101597240e-03
            4.57612493479132464990755124745192e-01 5.13261966319426554683502672560280e-01 1.75283779103676856832294550514462e-03
            3.74056887154247230675707669433905e-01 5.92330739629514768651574740943033e-01 1.96467115777034222884323000357654e-03
            2.94124419268578907171729497349588e-01 6.67970932566341102365470305812778e-01 2.08619573593893771826235372657266e-03
            2.20114584463026341509817029873375e-01 7.38006530515373793122080314788036e-01 2.09569589490422132291302936835109e-03
            1.54156478469823443600006385167944e-01 8.00422767559495262901236856123433e-01 1.98062109348444469539285428538733e-03
            9.81475205137383732711953143734718e-02 8.53424113546317641976202139630914e-01 1.73935089311050045286777265829414e-03
            5.36987667512220934185052101383917e-02 8.95486024048157913668433138809633e-01 1.38181088367753742048027465472160e-03
            2.20880252143010324950012090994278e-02 9.25399307748454802435844612773508e-01 9.28886974722132837713195474549366e-04
            4.21741578953443996269356830453034e-03 9.42310287486018527403075495385565e-01 4.11263091333856150525610795298803e-04
            9.95782584210465393503852737921989e-01 3.80348688681629489710678271308097e-03 2.30056297935333016710141192107120e-06
            9.77911974785698800971545097127091e-01 1.99201403064725515723587534466787e-02 2.77110190033184529444839488965258e-05
            9.46301233248777906581494789861608e-01 4.84283659399440749582232967895834e-02 1.03565716158896767859834420200116e-04
            9.01852479486261682239955916884355e-01 8.85145847307436811624015149391198e-02 2.50013541129748425272116829631841e-04
            8.45843521530176500888842383574229e-01 1.39026402336880794408102701709140e-01 4.76766411365518575672661771847061e-04
            7.79885415536973658490182970126625e-01 1.98510883769068419146819337584020e-01 7.81229763973354885200905961539775e-04
            7.05875580731421203850572965166066e-01 2.65256836794824535363801487619639e-01 1.14812844292922627831288195920934e-03
            6.25943112845752769324292330566095e-01 3.37344131148970660660779685713351e-01 1.55069061179292293044251760392171e-03
            5.42387506520867757053849800286116e-01 4.12698961888046111923245007346850e-01 1.95327063671890070045700227296948e-03
            4.57612493479132464990755124745192e-01 4.89153517598215292760954753248370e-01 2.31512383360814961649754017969371e-03
            3.74056887154247230675707669433905e-01 5.64508348337291021579176231171004e-01 2.59491040518148273485987687081433e-03
            2.94124419268578907171729497349588e-01 6.36595642691436980342700735491235e-01 2.75541838186133414972944777332486e-03
            2.20114584463026341509817029873375e-01 7.03341595717193235337560963671422e-01 2.76796606000705939371053787567689e-03
            1.54156478469823443600006385167944e-01 7.62826077149380887831853215175215e-01 2.61597685896577426822862122435254e-03
            9.81475205137383732711953143734718e-02 8.13337894755518098222069056646433e-01 2.29731052595914244859875985582676e-03
            5.36987667512220934185052101383917e-02 8.53424113546317641976202139630914e-01 1.82507664240216024159613183996953e-03
            2.20880252143010324950012090994278e-02 8.81932339179789304139944761118386e-01 1.22686102781673230487091696971902e-03
            4.21741578953443996269356830453034e-03 8.98048992599445483620002050884068e-01 5.43190584718744920965372013910155e-04
            9.95782584210465393503852737921989e-01 3.56727382317692130395125893471686e-03 2.79314703782445596466242274114045e-06
            9.77911974785698800971545097127091e-01 1.86830130309118096210152515368463e-02 3.36443519863876668789075263799049e-05
            9.46301233248777906581494789861608e-01 4.54207539706812518653933352652530e-02 1.25740645183599171951990824958045e-04
            9.01852479486261682239955916884355e-01 8.30174443807957102015038231002109e-02 3.03545083568567710102098500257739e-04
            8.45843521530176500888842383574229e-01 1.30392258615606299043676585824869e-01 5.78849047642294284830855044532427e-04
            7.79885415536973658490182970126625e-01 1.86182495262357639020933675055858e-01 9.48502440787708030363323263856046e-04
            7.05875580731421203850572965166066e-01 2.48783234562152733015949479522533e-01 1.39395947348122791961677613414849e-03
            6.25943112845752769324292330566095e-01 3.16393594683164303127398397919023e-01 1.88271606897242693538785385243273e-03
            5.42387506520867757053849800286116e-01 3.87068562980594110811694008589257e-01 2.37149434376904581431300123028905e-03
            4.57612493479132464990755124745192e-01 4.58774958549582168032543449953664e-01 2.81082558316098681666650449528788e-03
            3.74056887154247230675707669433905e-01 5.29449926847012197761443985655205e-01 3.15051853685390653714293307530170e-03
            2.94124419268578907171729497349588e-01 5.97060286968023601339439210278215e-01 3.34539361031811711619243787652067e-03
            2.20114584463026341509817029873375e-01 6.59661026267818861867908708518371e-01 3.36062792920390537160235666647168e-03
            1.54156478469823443600006385167944e-01 7.15451262914570174089590182120446e-01 3.17609562538099382544842796960438e-03
            9.81475205137383732711953143734718e-02 7.62826077149380887831853215175215e-01 2.78919818676270827445540234634791e-03
            5.36987667512220934185052101383917e-02 8.00422767559495262901236856123433e-01 2.21585214718230533170739704473817e-03
            2.20880252143010324950012090994278e-02 8.27160508499264812698470450413879e-01 1.48954985211133847204656444063176e-03
            4.21741578953443996269356830453034e-03 8.42276247706999714459641381836263e-01 6.59495604466247848512994167435863e-04
            9.95782584210465393503852737921989e-01 3.28910106551339028943536924032287e-03 3.20538270910144077821828274832416e-06
            9.77911974785698800971545097127091e-01 1.72261287226463978561330492311754e-02 3.86098628735589974379609212729036e-05
            9.46301233248777906581494789861608e-01 4.18788850215998653681026553385891e-02 1.44298486418636902383141884520512e-04
            9.01852479486261682239955916884355e-01 7.65438198197804925415610455274873e-02 3.48344769941391293761184799748776e-04
            8.45843521530176500888842383574229e-01 1.20224389269154782744486453793797e-01 6.64280363105272577545212797645036e-04
            7.79885415536973658490182970126625e-01 1.71664154169695548635132809067727e-01 1.08849025205973084368116232667489e-03
            7.05875580731421203850572965166066e-01 2.29383344940846584103510963359440e-01 1.59969150674039561205175274949397e-03
            6.25943112845752769324292330566095e-01 2.91721510872756939480865412406274e-01 2.16058283073135163135369829490173e-03
            5.42387506520867757053849800286116e-01 3.56885309631883640424376835653675e-01 2.72149903363838249759076859390916e-03
            4.57612493479132464990755124745192e-01 4.23000105905089796021201209441642e-01 3.22567039993060635308386174813222e-03
            3.74056887154247230675707669433905e-01 4.88163904664216719009317557720351e-01 3.61549804073356536790462278929681e-03
            2.94124419268578907171729497349588e-01 5.50502070596126880097642697364790e-01 3.83913438441978388873665473113306e-03
            2.20114584463026341509817029873375e-01 6.08221261367278054343898929801071e-01 3.85661710970426402020350131749638e-03
            1.54156478469823443600006385167944e-01 6.59661026267818861867908708518371e-01 3.64485000688631729723332419723647e-03
            9.81475205137383732711953143734718e-02 7.03341595717193235337560963671422e-01 3.20085105403898400808215818358349e-03
            5.36987667512220934185052101383917e-02 7.38006530515373793122080314788036e-01 2.54288587830149570567783712249366e-03
            2.20880252143010324950012090994278e-02 7.62659286814327375125799335364718e-01 1.70938990165771785047221786868477e-03
            4.21741578953443996269356830453034e-03 7.76596314471460380524092670384562e-01 7.56829403772108400168250330608544e-04
            9.95782584210465393503852737921989e-01 2.97697081962360575457804401366957e-03 3.52540747732508200133717465862837e-06
            9.77911974785698800971545097127091e-01 1.55913976253550939921055373815761e-02 4.24646638563474287182859800982015e-05
            9.46301233248777906581494789861608e-01 3.79046481650800251572697163737757e-02 1.58705218426027027354974219264250e-04
            9.01852479486261682239955916884355e-01 6.92799380399841541189331906025473e-02 3.83123442062469228824167499780629e-04
            8.45843521530176500888842383574229e-01 1.08815293763397449855467868928827e-01 7.30601981623603901304375707326244e-04
            7.79885415536973658490182970126625e-01 1.55373510135294129463900958398881e-01 1.19716490099946960487531377737014e-03
            7.05875580731421203850572965166066e-01 2.07615245258500030756110277252446e-01 1.75940438664715526756454622159254e-03
            6.25943112845752769324292330566095e-01 2.64037622446591957459105515226838e-01 2.37629498818122146183706711042305e-03
            5.42387506520867757053849800286116e-01 3.23017484584536107572461105519324e-01 2.99321295253736439218039677712113e-03
            4.57612493479132464990755124745192e-01 3.82858096146884874233506934615434e-01 3.54772068714671049219355758452821e-03
            3.74056887154247230675707669433905e-01 4.41837958284829246391467449939228e-01 3.97646864159611947847627888563693e-03
            2.94124419268578907171729497349588e-01 4.98260335472921034316584609769052e-01 4.22243279308243298075797866886205e-03
            2.20114584463026341509817029873375e-01 5.50502070596126991119945159880444e-01 4.24166099016071683480699761048527e-03
            1.54156478469823443600006385167944e-01 5.97060286968023712361741672793869e-01 4.00875110217572151044551631571267e-03
            9.81475205137383732711953143734718e-02 6.36595642691437091365003198006889e-01 3.52042338272805319793201661582316e-03
            5.36987667512220934185052101383917e-02 6.67970932566341213387772768328432e-01 2.79676709551525360872292758074309e-03
            2.20880252143010324950012090994278e-02 6.90284183106066207002982082485687e-01 1.88005504736045892140439139694763e-03
            4.21741578953443996269356830453034e-03 7.02898609911797644933528772526188e-01 8.32391099989937695811936801959519e-04
            9.95782584210465393503852737921989e-01 2.63986236746611979686605309325387e-03 3.74401370836179427179640777367808e-06
            9.77911974785698800971545097127091e-01 1.38258472592551334701127530024678e-02 4.50978460282195324233117661449910e-05
            9.46301233248777906581494789861608e-01 3.36123732162379659782480700869201e-02 1.68546336046931406619023285919923e-04
            9.01852479486261682239955916884355e-01 6.14347645084617755006917150240042e-02 4.06880460855268911903731554602359e-04
            8.45843521530176500888842383574229e-01 9.64931859987405576850605370964331e-02 7.75905722146635044335349906674537e-04
            7.79885415536973658490182970126625e-01 1.37779208181536050314974772845744e-01 1.27139964084733102754609657125684e-03
            7.05875580731421203850572965166066e-01 1.84105154560923495177249264997954e-01 1.86850291335880305289285363556928e-03
            6.25943112845752769324292330566095e-01 2.34138332326721987364237520523602e-01 2.52364615100109405607109458458126e-03
            5.42387506520867757053849800286116e-01 2.86439388645434722668881022400456e-01 3.17881844820078803687257718024739e-03
            4.57612493479132464990755124745192e-01 3.39503724200317880121957614392159e-01 3.76771053319326478317496942338494e-03
            3.74056887154247230675707669433905e-01 3.91804780519030781960054810042493e-01 4.22304462697258801290312035803254e-03
            2.94124419268578907171729497349588e-01 4.41837958284829190880316218681401e-01 4.48426071641852812960093999095079e-03
            2.20114584463026341509817029873375e-01 4.88163904664216719009317557720351e-01 4.50468123061758488751005202743727e-03
            1.54156478469823443600006385167944e-01 5.29449926847012197761443985655205e-01 4.25732888367966969683786615519239e-03
            9.81475205137383732711953143734718e-02 5.64508348337291021579176231171004e-01 3.73872055611909051217600641336958e-03
            5.36987667512220934185052101383917e-02 5.92330739629514768651574740943033e-01 2.97019122244822787043405654117123e-03
            2.20880252143010324950012090994278e-02 6.12117265586497705243118616635911e-01 1.99663497484074518861785207946014e-03
            4.21741578953443996269356830453034e-03 6.23303250478286718916365316545125e-01 8.84006659974898847606517016117778e-04
            9.95782584210465393503852737921989e-01 2.28747363404741216666438141658091e-03 3.85491207286117574273799288309128e-06
            9.77911974785698800971545097127091e-01 1.19802689199548523940608291127319e-02 4.64336523998160692595048304465166e-05
            9.46301233248777906581494789861608e-01 2.91255402014410288979995300451264e-02 1.73538709063147272328164749666257e-04
            9.01852479486261682239955916884355e-01 5.32339889226522769427596415425796e-02 4.18932333837151559940287226524447e-04
            8.45843521530176500888842383574229e-01 8.36125479712853669767014253011439e-02 7.98888190239526482620824321401187e-04
            7.79885415536973658490182970126625e-01 1.19387400615777766743619281442079e-01 1.30905872860124930859293534979315e-03
            7.05875580731421203850572965166066e-01 1.59529410373982688531313556268287e-01 1.92384830824638927183889425265306e-03
            6.25943112845752769324292330566095e-01 2.02883782320549765909589723378303e-01 2.59839700730702272829519827723743e-03
            5.42387506520867757053849800286116e-01 2.48203299290943368848871841692016e-01 3.27297562667441550160463137331135e-03
            4.57612493479132464990755124745192e-01 2.94184207229924221671524264820619e-01 3.87931080193828744032136235375674e-03
            3.74056887154247230675707669433905e-01 3.39503724200317991144260076907813e-01 4.34813197408705047475319105387825e-03
            2.94124419268578907171729497349588e-01 3.82858096146884985255809397131088e-01 4.61708533143769493667374348433441e-03
            2.20114584463026341509817029873375e-01 4.23000105905089962554654903215123e-01 4.63811070496775602195471677191563e-03
            1.54156478469823443600006385167944e-01 4.58774958549582334565997143727145e-01 4.38343173669049172846978734696677e-03
            9.81475205137383732711953143734718e-02 4.89153517598215514805559678279678e-01 3.84946213648986335781332357441897e-03
            5.36987667512220934185052101383917e-02 5.13261966319426776728107597591588e-01 3.05816882468944693795886990983490e-03
            2.20880252143010324950012090994278e-02 5.30407237600913017416814909665845e-01 2.05577566460840794118625218800389e-03
            4.21741578953443996269356830453034e-03 5.40100032886820402566740995098371e-01 9.10191097435378206753631147307715e-04
            9.95782584210465393503852737921989e-01 1.92994215548719541368505314693493e-03 3.85491207286117828383683464599230e-06
            9.77911974785698800971545097127091e-01 1.01077562943462963274132704327712e-02 4.64336523998161031408227206185302e-05
            9.46301233248777906581494789861608e-01 2.45732265497810749288465359541078e-02 1.73538709063147434958490622491922e-04
            9.01852479486261682239955916884355e-01 4.49135315910861240840112884598057e-02 4.18932333837151885200938972175777e-04
            8.45843521530176500888842383574229e-01 7.05439304985381043788805754957139e-02 7.98888190239527133142127812703848e-04
            7.79885415536973658490182970126625e-01 1.00727183847248574766197748431296e-01 1.30905872860125039279510783529759e-03
            7.05875580731421203850572965166066e-01 1.34595008894596107618113478565647e-01 1.92384830824639100656237022946016e-03
            6.25943112845752769324292330566095e-01 1.71173104833697548032844792942342e-01 2.59839700730702489669954324824630e-03
            5.42387506520867757053849800286116e-01 2.09409194188188929608429589279694e-01 3.27297562667441810368984533852199e-03
            4.57612493479132464990755124745192e-01 2.48203299290943424360023072949843e-01 3.87931080193829047608744531316916e-03
            3.74056887154247230675707669433905e-01 2.86439388645434889202334716173937e-01 4.34813197408705481156188099589599e-03
            2.94124419268578907171729497349588e-01 3.23017484584536274105914799292805e-01 4.61708533143769840612069543794860e-03
            2.20114584463026341509817029873375e-01 3.56885309631883862468981760684983e-01 4.63811070496775949140166872552982e-03
            1.54156478469823443600006385167944e-01 3.87068562980594332856298933620565e-01 4.38343173669049519791673930058096e-03
            9.81475205137383732711953143734718e-02 4.12698961888046389479001163635985e-01 3.84946213648986639357940653383139e-03
            5.36987667512220934185052101383917e-02 4.33039266929351407409143348559155e-01 3.05816882468944954004408387504554e-03
            2.20880252143010324950012090994278e-02 4.47504737184786227643940037523862e-01 2.05577566460840967590972816481099e-03
            4.21741578953443996269356830453034e-03 4.55682551323645324004019130370580e-01 9.10191097435378965695151887160819e-04
            9.95782584210465393503852737921989e-01 1.57755342206848669928120898475754e-03 3.74401370836181333003772099543571e-06
            9.77911974785698800971545097127091e-01 8.26217795504601004719091861261404e-03 4.50978460282197628162734193146832e-05
            9.46301233248777906581494789861608e-01 2.00863935349841274402571400514717e-02 1.68546336046932246875706962185859e-04
            9.01852479486261682239955916884355e-01 3.67127560052765977705035993494675e-02 4.06880460855270971887859277060784e-04
            8.45843521530176500888842383574229e-01 5.76632924710828928538397519787395e-02 7.75905722146639055883388103040943e-04
            7.79885415536973658490182970126625e-01 8.23353762814902495614788335842604e-02 1.27139964084733753275913148428344e-03
            7.05875580731421203850572965166066e-01 1.10019264707655259338814346392610e-01 1.86850291335881259387197150800830e-03
            6.25943112845752769324292330566095e-01 1.39918554827525243311470148910303e-01 2.52364615100110706649716441063447e-03
            5.42387506520867757053849800286116e-01 1.71173104833697437010542330426688e-01 3.17881844820080451674559895991479e-03
            4.57612493479132464990755124745192e-01 2.02883782320549682642862876491563e-01 3.76771053319328386513320516826298e-03
            3.74056887154247230675707669433905e-01 2.34138332326721987364237520523602e-01 4.22304462697260969694657006812122e-03
            2.94124419268578907171729497349588e-01 2.64037622446591901947954283969011e-01 4.48426071641855068100612768944302e-03
            2.20114584463026341509817029873375e-01 2.91721510872756939480865412406274e-01 4.50468123061760743891523972592950e-03
            1.54156478469823443600006385167944e-01 3.16393594683164303127398397919023e-01 4.25732888367969138088131586528107e-03
            9.81475205137383732711953143734718e-02 3.37344131148970660660779685713351e-01 3.73872055611910916045337316404584e-03
            5.36987667512220934185052101383917e-02 3.53970493619263082418768817660748e-01 2.97019122244824304926447133823331e-03
            2.20880252143010324950012090994278e-02 3.65794709199201262261880174264661e-01 1.99663497484075559695870794030270e-03
            4.21741578953443996269356830453034e-03 3.72479333732178785609789883892518e-01 8.84006659974903292835424206685957e-04
            9.95782584210465393503852737921989e-01 1.24044496991100139209052155564450e-03 3.52540747732509089518312082878193e-06
            9.77911974785698800971545097127091e-01 6.49662758894605212728334819871634e-03 4.24646638563475371385032286486449e-05
            9.46301233248777906581494789861608e-01 1.57941185861420717306824457182302e-02 1.58705218426027433930788901328413e-04
            9.01852479486261682239955916884355e-01 2.88675824737542330300499315853813e-02 3.83123442062470204606122736734619e-04
            8.45843521530176500888842383574229e-01 4.53411847064260076223263240535744e-02 7.30601981623605744448068932683782e-04
            7.79885415536973658490182970126625e-01 6.47410743277321704125526480311237e-02 1.19716490099947264064139673678255e-03
            7.05875580731421203850572965166066e-01 8.65091740100787376377411419525743e-02 1.75940438664715982121367066071116e-03
            6.25943112845752769324292330566095e-01 1.10019264707655314849965577650437e-01 2.37629498818122709968836403504611e-03
            5.42387506520867757053849800286116e-01 1.34595008894596107618113478565647e-01 2.99321295253737176475516967855128e-03
            4.57612493479132464990755124745192e-01 1.59529410373982716286889171897201e-01 3.54772068714671916581093746856368e-03
            3.74056887154247230675707669433905e-01 1.84105154560923578443976111884695e-01 3.97646864159612988681713474647950e-03
            2.94124419268578907171729497349588e-01 2.07615245258500114022837124139187e-01 4.22243279308244338909883452970462e-03
            2.20114584463026341509817029873375e-01 2.29383344940846722881389041504008e-01 4.24166099016072724314785347132783e-03
            1.54156478469823443600006385167944e-01 2.48783234562152871793827557667100e-01 4.00875110217573105142463418815169e-03
            9.81475205137383732711953143734718e-02 2.65256836794824701897255181393120e-01 3.52042338272806187154939649985863e-03
            5.36987667512220934185052101383917e-02 2.78330300682436859727175715306657e-01 2.79676709551526054761683148797147e-03
            2.20880252143010324950012090994278e-02 2.87627791679632927035470402188366e-01 1.88005504736046369189395033316714e-03
            4.21741578953443996269356830453034e-03 2.92883974298667915103777659169282e-01 8.32391099989939864216281772968387e-04
            9.95782584210465393503852737921989e-01 9.28314724021216098291675589138094e-04 3.20538270910145263667954430852891e-06
            9.77911974785698800971545097127091e-01 4.86189649165474479380888439550290e-03 3.86098628735591397394960599953606e-05
            9.46301233248777906581494789861608e-01 1.18198817296222263156790788229955e-02 1.44298486418637444484228127272729e-04
            9.01852479486261682239955916884355e-01 2.16037006939578772601873168923703e-02 3.48344769941392594803791782354097e-04
            8.45843521530176500888842383574229e-01 3.39320892006686539166260274669185e-02 6.64280363105275071210209514305234e-04
            7.79885415536973658490182970126625e-01 4.84504302933307443024268934550491e-02 1.08849025205973474680898327449086e-03
            7.05875580731421203850572965166066e-01 6.47410743277321426569770324022102e-02 1.59969150674040146674348417121791e-03
            6.25943112845752769324292330566095e-01 8.23353762814902634392666413987172e-02 2.16058283073135987129020918473543e-03
            5.42387506520867757053849800286116e-01 1.00727183847248533132834324987925e-01 2.72149903363839290593162445475173e-03
            4.57612493479132464990755124745192e-01 1.19387400615777752865831473627622e-01 3.22567039993061849614819358578188e-03
            3.74056887154247230675707669433905e-01 1.37779208181536078070550388474658e-01 3.61549804073357881201156160955179e-03
            2.94124419268578907171729497349588e-01 1.55373510135294157219476574027794e-01 3.83913438441979820020533153979159e-03
            2.20114584463026341509817029873375e-01 1.71664154169695576390708424696641e-01 3.85661710970427833167217812615490e-03
            1.54156478469823443600006385167944e-01 1.86182495262357666776509290684771e-01 3.64485000688633074134026301749145e-03
            9.81475205137383732711953143734718e-02 1.98510883769068474657970568841847e-01 3.20085105403899615114649002123315e-03
            5.36987667512220934185052101383917e-02 2.08294702733404113459414475073572e-01 2.54288587830150524665695499493268e-03
            2.20880252143010324950012090994278e-02 2.15252687971371620134775071164768e-01 1.70938990165772435568525278171137e-03
            4.21741578953443996269356830453034e-03 2.19186269739005151757638145681995e-01 7.56829403772111327514116041470515e-04
            9.95782584210465393503852737921989e-01 6.50141966357684975355568646193660e-04 2.79314703782444791784942382528723e-06
            9.77911974785698800971545097127091e-01 3.40501218338933172788407510722664e-03 3.36443519863875652349538558638642e-05
            9.46301233248777906581494789861608e-01 8.27801278054083981838839889633164e-03 1.25740645183598819586284767169104e-04
            9.01852479486261682239955916884355e-01 1.51300761329426648044149672500680e-02 3.03545083568566788530251887578970e-04
            8.45843521530176500888842383574229e-01 2.37642198542171376174358954358468e-02 5.78849047642292550107379067725333e-04
            7.79885415536973658490182970126625e-01 3.39320892006686469777321235596901e-02 9.48502440787705211437674801544517e-04
            7.05875580731421203850572965166066e-01 4.53411847064259798667507084246608e-02 1.39395947348122379964852068923165e-03
            6.25943112845752769324292330566095e-01 5.76632924710828928538397519787395e-02 1.88271606897242129753655692780967e-03
            5.42387506520867757053849800286116e-01 7.05439304985380627455171520523436e-02 2.37149434376903887541909732306067e-03
            4.57612493479132464990755124745192e-01 8.36125479712853392211258096722304e-02 2.81082558316097857672999360545418e-03
            3.74056887154247230675707669433905e-01 9.64931859987405576850605370964331e-02 3.15051853685389742984468419706445e-03
            2.94124419268578907171729497349588e-01 1.08815293763397435977680061114370e-01 3.34539361031810714153245100987988e-03
            2.20114584463026341509817029873375e-01 1.20224389269154782744486453793797e-01 3.36062792920389539694236979983089e-03
            1.54156478469823443600006385167944e-01 1.30392258615606299043676585824869e-01 3.17609562538098428446931009716536e-03
            9.81475205137383732711953143734718e-02 1.39026402336880794408102701709140e-01 2.78919818676270003451889145651421e-03
            5.36987667512220934185052101383917e-02 1.45878465689282615924682318109262e-01 2.21585214718229882649436213171157e-03
            2.20880252143010324950012090994278e-02 1.50751466286434127050952724857780e-01 1.48954985211133413523787449861402e-03
            4.21741578953443996269356830453034e-03 1.53506336503465762310938202972466e-01 6.59495604466245896949083693527882e-04
            9.95782584210465393503852737921989e-01 4.13928902718311924301131110581764e-04 2.30056297935330941479420419071289e-06
            9.77911974785698800971545097127091e-01 2.16788490782859194494491816840309e-03 2.77110190033181988345997726064240e-05
            9.46301233248777906581494789861608e-01 5.27040081127802279709060329082604e-03 1.03565716158895819182933495383736e-04
            9.01852479486261682239955916884355e-01 9.63293578299469557824075138796616e-03 2.50013541129746148447554610072530e-04
            8.45843521530176500888842383574229e-01 1.51300761329426648044149672500680e-02 4.76766411365514238863971829829325e-04
            7.79885415536973658490182970126625e-01 2.16037006939578703212934129851419e-02 7.81229763973347837886784805760954e-04
            7.05875580731421203850572965166066e-01 2.88675824737542191522621237709245e-02 1.14812844292921586997202609836677e-03
            6.25943112845752769324292330566095e-01 3.67127560052765977705035993494675e-02 1.55069061179290883581427529236407e-03
            5.42387506520867757053849800286116e-01 4.49135315910860963284356728308921e-02 1.95327063671888291954137351069676e-03
            4.57612493479132464990755124745192e-01 5.32339889226522491871840259136661e-02 2.31512383360812836613495946380681e-03
            3.74056887154247230675707669433905e-01 6.14347645084617755006917150240042e-02 2.59491040518145931609295118391856e-03
            2.94124419268578907171729497349588e-01 6.92799380399841402411453827880905e-02 2.75541838186130899623904610962200e-03
            2.20114584463026341509817029873375e-01 7.65438198197804925415610455274873e-02 2.76796606000703467390100520617580e-03
            1.54156478469823443600006385167944e-01 8.30174443807957102015038231002109e-02 2.61597685896575041578082654325499e-03
            9.81475205137383732711953143734718e-02 8.85145847307436811624015149391198e-02 2.29731052595912163191704813414162e-03
            5.36987667512220934185052101383917e-02 9.28771197024603478720194971174351e-02 1.82507664240214376172311006030213e-03
            2.20880252143010324950012090994278e-02 9.59796356059098021429321079267538e-02 1.22686102781672102916832312047291e-03
            4.21741578953443996269356830453034e-03 9.77335916110200764173043808114016e-02 5.43190584718739933635378580589759e-04
            9.95782584210465393503852737921989e-01 2.26470026775140012348028206723427e-04 1.74181340640683852540667071739922e-06
            9.77911974785698800971545097127091e-01 1.18609971397786958559328329698701e-03 2.09807011754758989589917522744500e-05
            9.46301233248777906581494789861608e-01 2.88355755060215558291392135004116e-03 7.84121775707259937053181242383459e-05
            9.01852479486261682239955916884355e-01 5.27040081127802279709060329082604e-03 1.89291465450536146482296562254533e-04
            8.45843521530176500888842383574229e-01 8.27801278054083981838839889633164e-03 3.60971698881448342828559550099499e-04
            7.79885415536973658490182970126625e-01 1.18198817296222228462321268693813e-02 5.91488469815913318027023048983892e-04
            7.05875580731421203850572965166066e-01 1.57941185861420613223415898573876e-02 8.69276578002341942033592836480693e-04
            6.25943112845752769324292330566095e-01 2.00863935349841274402571400514717e-02 1.17406640072482124949326820484430e-03
            5.42387506520867757053849800286116e-01 2.45732265497810610510587281396511e-02 1.47886974271582141357561646088925e-03
            4.57612493479132464990755124745192e-01 2.91255402014410184896586741842839e-02 1.75283779103677355565293893846501e-03
            3.74056887154247230675707669433905e-01 3.36123732162379659782480700869201e-02 1.96467115777034786669452692819959e-03
            2.94124419268578907171729497349588e-01 3.79046481650800182183758124665474e-02 2.08619573593894378979451964539749e-03
            2.20114584463026341509817029873375e-01 4.18788850215998653681026553385891e-02 2.09569589490422739444519528717592e-03
            1.54156478469823443600006385167944e-01 4.54207539706812518653933352652530e-02 1.98062109348445033324415121001039e-03
            9.81475205137383732711953143734718e-02 4.84283659399440749582232967895834e-02 1.73935089311050544019776609161454e-03
            5.36987667512220934185052101383917e-02 5.08152092006199374019104197941488e-02 1.38181088367754132360809560253756e-03
            2.20880252143010324950012090994278e-02 5.25126670372442344580932171993481e-02 9.28886974722135548218626688310451e-04
            4.21741578953443996269356830453034e-03 5.34722967244469563063979933303926e-02 4.11263091333857343148000529353681e-04
            9.95782584210465393503852737921989e-01 9.31543862984316882631941059145220e-05 1.13304076349583715411085786672896e-06
            9.77911974785698800971545097127091e-01 4.87880857867600630471682787714371e-04 1.36478394247625573772577170506004e-05
            9.46301233248777906581494789861608e-01 1.18609971397786351406111737816218e-03 5.10067227725511117318937803766232e-05
            9.01852479486261682239955916884355e-01 2.16788490782858110292319331335875e-03 1.23133135701233685380465154857177e-04
            8.45843521530176500888842383574229e-01 3.40501218338931438064931533915569e-03 2.34810254529351002204112797144830e-04
            7.79885415536973658490182970126625e-01 4.86189649165471877295674474339648e-03 3.84760241811269788122368273874940e-04
            7.05875580731421203850572965166066e-01 6.49662758894601569809035268576736e-03 5.65459993594038894636211178834628e-04
            6.25943112845752769324292330566095e-01 8.26217795504596841382749516924378e-03 7.63724223374905183091276850149143e-04
            5.42387506520867757053849800286116e-01 1.01077562943462390815385631981371e-02 9.61997247371196902848700105437274e-04
            4.57612493479132464990755124745192e-01 1.19802689199547882092922179708694e-02 1.14021206963702752094913517311170e-03
            3.74056887154247230675707669433905e-01 1.38258472592550640811737139301840e-02 1.27800859749406590763765034779453e-03
            2.94124419268578907171729497349588e-01 1.55913976253550124601021664716427e-02 1.35705971762277053868639420386444e-03
            2.20114584463026341509817029873375e-01 1.72261287226463111199592503908207e-02 1.36323952272001205389406752743753e-03
            1.54156478469823443600006385167944e-01 1.86830130309117159459475487892632e-02 1.28838394956837134879001904863571e-03
            9.81475205137383732711953143734718e-02 1.99201403064724544278440987454815e-02 1.13143891111880693853275481330911e-03
            5.36987667512220934185052101383917e-02 2.09019255003231700651422642067701e-02 8.98860953125060237189902778709438e-04
            2.20880252143010324950012090994278e-02 2.16001443564334365265988679993825e-02 6.04236253532819704223366663597972e-04
            4.21741578953443996269356830453034e-03 2.19948708280026036798737720801000e-02 2.67524549580675668515594711749372e-04
            9.95782584210465393503852737921989e-01 1.77865959418151070917362560619424e-05 4.92649033616421286018165377751377e-07
            9.77911974785698800971545097127091e-01 9.31543862984284763142581176076362e-05 5.93411562953539969354202140983112e-06
            9.46301233248777906581494789861608e-01 2.26470026775131067680105201311846e-04 2.21778540467580794967594748356987e-05
            9.01852479486261682239955916884355e-01 4.13928902718295552848326579464810e-04 5.35386036087617838601440678480969e-05
            8.45843521530176500888842383574229e-01 6.50141966357659279764080739738574e-04 1.02096101662043766355146556801259e-04
            7.79885415536973658490182970126625e-01 9.28314724021179126997593833436895e-04 1.67294741203756426508458821089675e-04
            7.05875580731421203850572965166066e-01 1.24044496991095173563102171954142e-03 2.45863457315827216038278679732798e-04
            6.25943112845752769324292330566095e-01 1.57755342206842446607650831680303e-03 3.32069253567046369490095569787513e-04
            5.42387506520867757053849800286116e-01 1.92994215548711800164993768191835e-03 4.18278873565715921412772049237105e-04
            4.57612493479132464990755124745192e-01 2.28747363404732066000102364000668e-03 4.95767135942523553410188341672438e-04
            3.74056887154247230675707669433905e-01 2.63986236746601527977662549062643e-03 5.55681420116217023551230536071444e-04
            2.94124419268578907171729497349588e-01 2.97697081962348779338167759078715e-03 5.90053050151436819618921969521352e-04
            2.20114584463026341509817029873375e-01 3.28910106551326061885553997399256e-03 5.92740045277455633240815124906931e-04
            1.54156478469823443600006385167944e-01 3.56727382317678035766883581914044e-03 5.60192649842026807634198792840152e-04
            9.81475205137383732711953143734718e-02 3.80348688681614484352611071926731e-03 4.91952544089332001291114959684592e-04
            5.36987667512220934185052101383917e-02 3.99094576275930892211851741535611e-03 3.90827050693506037378627260281405e-04
            2.20880252143010324950012090994278e-02 4.12426140323601201492786927360612e-03 2.62723474714636115717042441630724e-04
            4.21741578953443996269356830453034e-03 4.19962919359262529944709640972178e-03 1.16320361160662233131028731047962e-04
            ];
        
    case  34
        
        % ALG. DEG.:   34
        % PTS CARD.:  324
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.2e-15
        
        xyw=[
            9.95782584210465393503852737921989e-01 4.19962919359279096553905219479930e-03 4.92649033616424885908191208527818e-07
            9.77911974785698800971545097127091e-01 2.19948708280027112327292826421399e-02 5.93411562953544289222233137914841e-06
            9.46301233248777906581494789861608e-01 5.34722967244469563063979933303926e-02 2.21778540467582421270853476613638e-05
            9.01852479486261682239955916884355e-01 9.77335916110200625395165729969449e-02 5.35386036087621701071680158090516e-05
            8.45843521530176500888842383574229e-01 1.53506336503465762310938202972466e-01 1.02096101662044511744140140585557e-04
            7.79885415536973658490182970126625e-01 2.19186269739005068490911298795254e-01 1.67294741203757619130848555144553e-04
            7.05875580731421203850572965166066e-01 2.92883974298667748570323965395801e-01 2.45863457315829004971863280815114e-04
            6.25943112845752769324292330566095e-01 3.72479333732178730098638652634691e-01 3.32069253567048754734875037897268e-04
            5.42387506520867757053849800286116e-01 4.55682551323644990937111742823618e-01 4.18278873565718957178855008649521e-04
            4.57612493479132464990755124745192e-01 5.40100032886820069499833607551409e-01 4.95767135942527131277357543837070e-04
            3.74056887154247230675707669433905e-01 6.23303250478286607894062854029471e-01 5.55681420116221035099268732437849e-04
            2.94124419268578907171729497349588e-01 7.02898609911797533911226310010534e-01 5.90053050151441156427611911539088e-04
            2.20114584463026341509817029873375e-01 7.76596314471460269501790207868908e-01 5.92740045277459861629287818374223e-04
            1.54156478469823443600006385167944e-01 8.42276247706999603437338919320609e-01 5.60192649842030819182236989206558e-04
            9.81475205137383732711953143734718e-02 8.98048992599445372597699588368414e-01 4.91952544089335579158284161849224e-04
            5.36987667512220934185052101383917e-02 9.42310287486018416380773032869911e-01 3.90827050693508856304275722592934e-04
            2.20880252143010324950012090994278e-02 9.73787713382462838396236293192487e-01 2.62723474714638013070844291263484e-04
            4.21741578953443996269356830453034e-03 9.91582955016872658049464916985016e-01 1.16320361160663073387712407313899e-04
            9.95782584210465393503852737921989e-01 4.12426140323617421157287310506945e-03 1.13304076349582783674843806942523e-06
            9.77911974785698800971545097127091e-01 2.16001443564335406100074266078082e-02 1.36478394247624472629745739915563e-05
            9.46301233248777906581494789861608e-01 5.25126670372442205803054093848914e-02 5.10067227725507051560790983124605e-05
            9.01852479486261682239955916884355e-01 9.59796356059097743873564922978403e-02 1.23133135701232682493455605765575e-04
            8.45843521530176500888842383574229e-01 1.50751466286434099295377109228866e-01 2.34810254529349104850310947512071e-04
            7.79885415536973658490182970126625e-01 2.15252687971371536868048224278027e-01 3.84760241811266698146176690187303e-04
            7.05875580731421203850572965166066e-01 2.87627791679632649479714245899231e-01 5.65459993594034340987086739716005e-04
            6.25943112845752769324292330566095e-01 3.65794709199201206750728943006834e-01 7.63724223374899003138893682773869e-04
            5.42387506520867757053849800286116e-01 4.47504737184785894577032649976900e-01 9.61997247371189096593058209805349e-04
            4.57612493479132464990755124745192e-01 5.30407237600912684349907522118883e-01 1.14021206963701841365088629487445e-03
            3.74056887154247230675707669433905e-01 6.12117265586497594220816154120257e-01 1.27800859749405571613722898405285e-03
            2.94124419268578907171729497349588e-01 6.90284183106065984958377157454379e-01 1.35705971762275947982423485171921e-03
            2.20114584463026341509817029873375e-01 7.62659286814327264103496872849064e-01 1.36323952272000099503190817529230e-03
            1.54156478469823443600006385167944e-01 8.27160508499264590653865525382571e-01 1.28838394956836072360872869069226e-03
            9.81475205137383732711953143734718e-02 8.81932339179789082095339836087078e-01 1.13143891111879761439407143797098e-03
            5.36987667512220934185052101383917e-02 9.25399307748454580391239687742200e-01 8.98860953125052973035347125829730e-04
            2.20880252143010324950012090994278e-02 9.56311830429265374853287084988551e-01 6.04236253532814825313590478828019e-04
            4.21741578953443996269356830453034e-03 9.73787713382462727373933830676833e-01 2.67524549580673500111249740740504e-04
            9.95782584210465393503852737921989e-01 3.99094576275946678195483130480170e-03 1.74181340640683344320898719159718e-06
            9.77911974785698800971545097127091e-01 2.09019255003232741485508228151957e-02 2.09807011754758379726195499648256e-05
            9.46301233248777906581494789861608e-01 5.08152092006199374019104197941488e-02 7.84121775707257633123564710686537e-05
            9.01852479486261682239955916884355e-01 9.28771197024603478720194971174351e-02 1.89291465450535604381210319502316e-04
            8.45843521530176500888842383574229e-01 1.45878465689282615924682318109262e-01 3.60971698881447312836495688870286e-04
            7.79885415536973658490182970126625e-01 2.08294702733404057948263243815745e-01 5.91488469815911583303547072176798e-04
            7.05875580731421203850572965166066e-01 2.78330300682436693193722021533176e-01 8.69276578002339339948378871270052e-04
            6.25943112845752769324292330566095e-01 3.53970493619263082418768817660748e-01 1.17406640072481799688675074833100e-03
            5.42387506520867757053849800286116e-01 4.33039266929351129853387192270020e-01 1.47886974271581729360736101597240e-03
            4.57612493479132464990755124745192e-01 5.13261966319426554683502672560280e-01 1.75283779103676856832294550514462e-03
            3.74056887154247230675707669433905e-01 5.92330739629514768651574740943033e-01 1.96467115777034222884323000357654e-03
            2.94124419268578907171729497349588e-01 6.67970932566341102365470305812778e-01 2.08619573593893771826235372657266e-03
            2.20114584463026341509817029873375e-01 7.38006530515373793122080314788036e-01 2.09569589490422132291302936835109e-03
            1.54156478469823443600006385167944e-01 8.00422767559495262901236856123433e-01 1.98062109348444469539285428538733e-03
            9.81475205137383732711953143734718e-02 8.53424113546317641976202139630914e-01 1.73935089311050045286777265829414e-03
            5.36987667512220934185052101383917e-02 8.95486024048157913668433138809633e-01 1.38181088367753742048027465472160e-03
            2.20880252143010324950012090994278e-02 9.25399307748454802435844612773508e-01 9.28886974722132837713195474549366e-04
            4.21741578953443996269356830453034e-03 9.42310287486018527403075495385565e-01 4.11263091333856150525610795298803e-04
            9.95782584210465393503852737921989e-01 3.80348688681629489710678271308097e-03 2.30056297935333016710141192107120e-06
            9.77911974785698800971545097127091e-01 1.99201403064725515723587534466787e-02 2.77110190033184529444839488965258e-05
            9.46301233248777906581494789861608e-01 4.84283659399440749582232967895834e-02 1.03565716158896767859834420200116e-04
            9.01852479486261682239955916884355e-01 8.85145847307436811624015149391198e-02 2.50013541129748425272116829631841e-04
            8.45843521530176500888842383574229e-01 1.39026402336880794408102701709140e-01 4.76766411365518575672661771847061e-04
            7.79885415536973658490182970126625e-01 1.98510883769068419146819337584020e-01 7.81229763973354885200905961539775e-04
            7.05875580731421203850572965166066e-01 2.65256836794824535363801487619639e-01 1.14812844292922627831288195920934e-03
            6.25943112845752769324292330566095e-01 3.37344131148970660660779685713351e-01 1.55069061179292293044251760392171e-03
            5.42387506520867757053849800286116e-01 4.12698961888046111923245007346850e-01 1.95327063671890070045700227296948e-03
            4.57612493479132464990755124745192e-01 4.89153517598215292760954753248370e-01 2.31512383360814961649754017969371e-03
            3.74056887154247230675707669433905e-01 5.64508348337291021579176231171004e-01 2.59491040518148273485987687081433e-03
            2.94124419268578907171729497349588e-01 6.36595642691436980342700735491235e-01 2.75541838186133414972944777332486e-03
            2.20114584463026341509817029873375e-01 7.03341595717193235337560963671422e-01 2.76796606000705939371053787567689e-03
            1.54156478469823443600006385167944e-01 7.62826077149380887831853215175215e-01 2.61597685896577426822862122435254e-03
            9.81475205137383732711953143734718e-02 8.13337894755518098222069056646433e-01 2.29731052595914244859875985582676e-03
            5.36987667512220934185052101383917e-02 8.53424113546317641976202139630914e-01 1.82507664240216024159613183996953e-03
            2.20880252143010324950012090994278e-02 8.81932339179789304139944761118386e-01 1.22686102781673230487091696971902e-03
            4.21741578953443996269356830453034e-03 8.98048992599445483620002050884068e-01 5.43190584718744920965372013910155e-04
            9.95782584210465393503852737921989e-01 3.56727382317692130395125893471686e-03 2.79314703782445596466242274114045e-06
            9.77911974785698800971545097127091e-01 1.86830130309118096210152515368463e-02 3.36443519863876668789075263799049e-05
            9.46301233248777906581494789861608e-01 4.54207539706812518653933352652530e-02 1.25740645183599171951990824958045e-04
            9.01852479486261682239955916884355e-01 8.30174443807957102015038231002109e-02 3.03545083568567710102098500257739e-04
            8.45843521530176500888842383574229e-01 1.30392258615606299043676585824869e-01 5.78849047642294284830855044532427e-04
            7.79885415536973658490182970126625e-01 1.86182495262357639020933675055858e-01 9.48502440787708030363323263856046e-04
            7.05875580731421203850572965166066e-01 2.48783234562152733015949479522533e-01 1.39395947348122791961677613414849e-03
            6.25943112845752769324292330566095e-01 3.16393594683164303127398397919023e-01 1.88271606897242693538785385243273e-03
            5.42387506520867757053849800286116e-01 3.87068562980594110811694008589257e-01 2.37149434376904581431300123028905e-03
            4.57612493479132464990755124745192e-01 4.58774958549582168032543449953664e-01 2.81082558316098681666650449528788e-03
            3.74056887154247230675707669433905e-01 5.29449926847012197761443985655205e-01 3.15051853685390653714293307530170e-03
            2.94124419268578907171729497349588e-01 5.97060286968023601339439210278215e-01 3.34539361031811711619243787652067e-03
            2.20114584463026341509817029873375e-01 6.59661026267818861867908708518371e-01 3.36062792920390537160235666647168e-03
            1.54156478469823443600006385167944e-01 7.15451262914570174089590182120446e-01 3.17609562538099382544842796960438e-03
            9.81475205137383732711953143734718e-02 7.62826077149380887831853215175215e-01 2.78919818676270827445540234634791e-03
            5.36987667512220934185052101383917e-02 8.00422767559495262901236856123433e-01 2.21585214718230533170739704473817e-03
            2.20880252143010324950012090994278e-02 8.27160508499264812698470450413879e-01 1.48954985211133847204656444063176e-03
            4.21741578953443996269356830453034e-03 8.42276247706999714459641381836263e-01 6.59495604466247848512994167435863e-04
            9.95782584210465393503852737921989e-01 3.28910106551339028943536924032287e-03 3.20538270910144077821828274832416e-06
            9.77911974785698800971545097127091e-01 1.72261287226463978561330492311754e-02 3.86098628735589974379609212729036e-05
            9.46301233248777906581494789861608e-01 4.18788850215998653681026553385891e-02 1.44298486418636902383141884520512e-04
            9.01852479486261682239955916884355e-01 7.65438198197804925415610455274873e-02 3.48344769941391293761184799748776e-04
            8.45843521530176500888842383574229e-01 1.20224389269154782744486453793797e-01 6.64280363105272577545212797645036e-04
            7.79885415536973658490182970126625e-01 1.71664154169695548635132809067727e-01 1.08849025205973084368116232667489e-03
            7.05875580731421203850572965166066e-01 2.29383344940846584103510963359440e-01 1.59969150674039561205175274949397e-03
            6.25943112845752769324292330566095e-01 2.91721510872756939480865412406274e-01 2.16058283073135163135369829490173e-03
            5.42387506520867757053849800286116e-01 3.56885309631883640424376835653675e-01 2.72149903363838249759076859390916e-03
            4.57612493479132464990755124745192e-01 4.23000105905089796021201209441642e-01 3.22567039993060635308386174813222e-03
            3.74056887154247230675707669433905e-01 4.88163904664216719009317557720351e-01 3.61549804073356536790462278929681e-03
            2.94124419268578907171729497349588e-01 5.50502070596126880097642697364790e-01 3.83913438441978388873665473113306e-03
            2.20114584463026341509817029873375e-01 6.08221261367278054343898929801071e-01 3.85661710970426402020350131749638e-03
            1.54156478469823443600006385167944e-01 6.59661026267818861867908708518371e-01 3.64485000688631729723332419723647e-03
            9.81475205137383732711953143734718e-02 7.03341595717193235337560963671422e-01 3.20085105403898400808215818358349e-03
            5.36987667512220934185052101383917e-02 7.38006530515373793122080314788036e-01 2.54288587830149570567783712249366e-03
            2.20880252143010324950012090994278e-02 7.62659286814327375125799335364718e-01 1.70938990165771785047221786868477e-03
            4.21741578953443996269356830453034e-03 7.76596314471460380524092670384562e-01 7.56829403772108400168250330608544e-04
            9.95782584210465393503852737921989e-01 2.97697081962360575457804401366957e-03 3.52540747732508200133717465862837e-06
            9.77911974785698800971545097127091e-01 1.55913976253550939921055373815761e-02 4.24646638563474287182859800982015e-05
            9.46301233248777906581494789861608e-01 3.79046481650800251572697163737757e-02 1.58705218426027027354974219264250e-04
            9.01852479486261682239955916884355e-01 6.92799380399841541189331906025473e-02 3.83123442062469228824167499780629e-04
            8.45843521530176500888842383574229e-01 1.08815293763397449855467868928827e-01 7.30601981623603901304375707326244e-04
            7.79885415536973658490182970126625e-01 1.55373510135294129463900958398881e-01 1.19716490099946960487531377737014e-03
            7.05875580731421203850572965166066e-01 2.07615245258500030756110277252446e-01 1.75940438664715526756454622159254e-03
            6.25943112845752769324292330566095e-01 2.64037622446591957459105515226838e-01 2.37629498818122146183706711042305e-03
            5.42387506520867757053849800286116e-01 3.23017484584536107572461105519324e-01 2.99321295253736439218039677712113e-03
            4.57612493479132464990755124745192e-01 3.82858096146884874233506934615434e-01 3.54772068714671049219355758452821e-03
            3.74056887154247230675707669433905e-01 4.41837958284829246391467449939228e-01 3.97646864159611947847627888563693e-03
            2.94124419268578907171729497349588e-01 4.98260335472921034316584609769052e-01 4.22243279308243298075797866886205e-03
            2.20114584463026341509817029873375e-01 5.50502070596126991119945159880444e-01 4.24166099016071683480699761048527e-03
            1.54156478469823443600006385167944e-01 5.97060286968023712361741672793869e-01 4.00875110217572151044551631571267e-03
            9.81475205137383732711953143734718e-02 6.36595642691437091365003198006889e-01 3.52042338272805319793201661582316e-03
            5.36987667512220934185052101383917e-02 6.67970932566341213387772768328432e-01 2.79676709551525360872292758074309e-03
            2.20880252143010324950012090994278e-02 6.90284183106066207002982082485687e-01 1.88005504736045892140439139694763e-03
            4.21741578953443996269356830453034e-03 7.02898609911797644933528772526188e-01 8.32391099989937695811936801959519e-04
            9.95782584210465393503852737921989e-01 2.63986236746611979686605309325387e-03 3.74401370836179427179640777367808e-06
            9.77911974785698800971545097127091e-01 1.38258472592551334701127530024678e-02 4.50978460282195324233117661449910e-05
            9.46301233248777906581494789861608e-01 3.36123732162379659782480700869201e-02 1.68546336046931406619023285919923e-04
            9.01852479486261682239955916884355e-01 6.14347645084617755006917150240042e-02 4.06880460855268911903731554602359e-04
            8.45843521530176500888842383574229e-01 9.64931859987405576850605370964331e-02 7.75905722146635044335349906674537e-04
            7.79885415536973658490182970126625e-01 1.37779208181536050314974772845744e-01 1.27139964084733102754609657125684e-03
            7.05875580731421203850572965166066e-01 1.84105154560923495177249264997954e-01 1.86850291335880305289285363556928e-03
            6.25943112845752769324292330566095e-01 2.34138332326721987364237520523602e-01 2.52364615100109405607109458458126e-03
            5.42387506520867757053849800286116e-01 2.86439388645434722668881022400456e-01 3.17881844820078803687257718024739e-03
            4.57612493479132464990755124745192e-01 3.39503724200317880121957614392159e-01 3.76771053319326478317496942338494e-03
            3.74056887154247230675707669433905e-01 3.91804780519030781960054810042493e-01 4.22304462697258801290312035803254e-03
            2.94124419268578907171729497349588e-01 4.41837958284829190880316218681401e-01 4.48426071641852812960093999095079e-03
            2.20114584463026341509817029873375e-01 4.88163904664216719009317557720351e-01 4.50468123061758488751005202743727e-03
            1.54156478469823443600006385167944e-01 5.29449926847012197761443985655205e-01 4.25732888367966969683786615519239e-03
            9.81475205137383732711953143734718e-02 5.64508348337291021579176231171004e-01 3.73872055611909051217600641336958e-03
            5.36987667512220934185052101383917e-02 5.92330739629514768651574740943033e-01 2.97019122244822787043405654117123e-03
            2.20880252143010324950012090994278e-02 6.12117265586497705243118616635911e-01 1.99663497484074518861785207946014e-03
            4.21741578953443996269356830453034e-03 6.23303250478286718916365316545125e-01 8.84006659974898847606517016117778e-04
            9.95782584210465393503852737921989e-01 2.28747363404741216666438141658091e-03 3.85491207286117574273799288309128e-06
            9.77911974785698800971545097127091e-01 1.19802689199548523940608291127319e-02 4.64336523998160692595048304465166e-05
            9.46301233248777906581494789861608e-01 2.91255402014410288979995300451264e-02 1.73538709063147272328164749666257e-04
            9.01852479486261682239955916884355e-01 5.32339889226522769427596415425796e-02 4.18932333837151559940287226524447e-04
            8.45843521530176500888842383574229e-01 8.36125479712853669767014253011439e-02 7.98888190239526482620824321401187e-04
            7.79885415536973658490182970126625e-01 1.19387400615777766743619281442079e-01 1.30905872860124930859293534979315e-03
            7.05875580731421203850572965166066e-01 1.59529410373982688531313556268287e-01 1.92384830824638927183889425265306e-03
            6.25943112845752769324292330566095e-01 2.02883782320549765909589723378303e-01 2.59839700730702272829519827723743e-03
            5.42387506520867757053849800286116e-01 2.48203299290943368848871841692016e-01 3.27297562667441550160463137331135e-03
            4.57612493479132464990755124745192e-01 2.94184207229924221671524264820619e-01 3.87931080193828744032136235375674e-03
            3.74056887154247230675707669433905e-01 3.39503724200317991144260076907813e-01 4.34813197408705047475319105387825e-03
            2.94124419268578907171729497349588e-01 3.82858096146884985255809397131088e-01 4.61708533143769493667374348433441e-03
            2.20114584463026341509817029873375e-01 4.23000105905089962554654903215123e-01 4.63811070496775602195471677191563e-03
            1.54156478469823443600006385167944e-01 4.58774958549582334565997143727145e-01 4.38343173669049172846978734696677e-03
            9.81475205137383732711953143734718e-02 4.89153517598215514805559678279678e-01 3.84946213648986335781332357441897e-03
            5.36987667512220934185052101383917e-02 5.13261966319426776728107597591588e-01 3.05816882468944693795886990983490e-03
            2.20880252143010324950012090994278e-02 5.30407237600913017416814909665845e-01 2.05577566460840794118625218800389e-03
            4.21741578953443996269356830453034e-03 5.40100032886820402566740995098371e-01 9.10191097435378206753631147307715e-04
            9.95782584210465393503852737921989e-01 1.92994215548719541368505314693493e-03 3.85491207286117828383683464599230e-06
            9.77911974785698800971545097127091e-01 1.01077562943462963274132704327712e-02 4.64336523998161031408227206185302e-05
            9.46301233248777906581494789861608e-01 2.45732265497810749288465359541078e-02 1.73538709063147434958490622491922e-04
            9.01852479486261682239955916884355e-01 4.49135315910861240840112884598057e-02 4.18932333837151885200938972175777e-04
            8.45843521530176500888842383574229e-01 7.05439304985381043788805754957139e-02 7.98888190239527133142127812703848e-04
            7.79885415536973658490182970126625e-01 1.00727183847248574766197748431296e-01 1.30905872860125039279510783529759e-03
            7.05875580731421203850572965166066e-01 1.34595008894596107618113478565647e-01 1.92384830824639100656237022946016e-03
            6.25943112845752769324292330566095e-01 1.71173104833697548032844792942342e-01 2.59839700730702489669954324824630e-03
            5.42387506520867757053849800286116e-01 2.09409194188188929608429589279694e-01 3.27297562667441810368984533852199e-03
            4.57612493479132464990755124745192e-01 2.48203299290943424360023072949843e-01 3.87931080193829047608744531316916e-03
            3.74056887154247230675707669433905e-01 2.86439388645434889202334716173937e-01 4.34813197408705481156188099589599e-03
            2.94124419268578907171729497349588e-01 3.23017484584536274105914799292805e-01 4.61708533143769840612069543794860e-03
            2.20114584463026341509817029873375e-01 3.56885309631883862468981760684983e-01 4.63811070496775949140166872552982e-03
            1.54156478469823443600006385167944e-01 3.87068562980594332856298933620565e-01 4.38343173669049519791673930058096e-03
            9.81475205137383732711953143734718e-02 4.12698961888046389479001163635985e-01 3.84946213648986639357940653383139e-03
            5.36987667512220934185052101383917e-02 4.33039266929351407409143348559155e-01 3.05816882468944954004408387504554e-03
            2.20880252143010324950012090994278e-02 4.47504737184786227643940037523862e-01 2.05577566460840967590972816481099e-03
            4.21741578953443996269356830453034e-03 4.55682551323645324004019130370580e-01 9.10191097435378965695151887160819e-04
            9.95782584210465393503852737921989e-01 1.57755342206848669928120898475754e-03 3.74401370836181333003772099543571e-06
            9.77911974785698800971545097127091e-01 8.26217795504601004719091861261404e-03 4.50978460282197628162734193146832e-05
            9.46301233248777906581494789861608e-01 2.00863935349841274402571400514717e-02 1.68546336046932246875706962185859e-04
            9.01852479486261682239955916884355e-01 3.67127560052765977705035993494675e-02 4.06880460855270971887859277060784e-04
            8.45843521530176500888842383574229e-01 5.76632924710828928538397519787395e-02 7.75905722146639055883388103040943e-04
            7.79885415536973658490182970126625e-01 8.23353762814902495614788335842604e-02 1.27139964084733753275913148428344e-03
            7.05875580731421203850572965166066e-01 1.10019264707655259338814346392610e-01 1.86850291335881259387197150800830e-03
            6.25943112845752769324292330566095e-01 1.39918554827525243311470148910303e-01 2.52364615100110706649716441063447e-03
            5.42387506520867757053849800286116e-01 1.71173104833697437010542330426688e-01 3.17881844820080451674559895991479e-03
            4.57612493479132464990755124745192e-01 2.02883782320549682642862876491563e-01 3.76771053319328386513320516826298e-03
            3.74056887154247230675707669433905e-01 2.34138332326721987364237520523602e-01 4.22304462697260969694657006812122e-03
            2.94124419268578907171729497349588e-01 2.64037622446591901947954283969011e-01 4.48426071641855068100612768944302e-03
            2.20114584463026341509817029873375e-01 2.91721510872756939480865412406274e-01 4.50468123061760743891523972592950e-03
            1.54156478469823443600006385167944e-01 3.16393594683164303127398397919023e-01 4.25732888367969138088131586528107e-03
            9.81475205137383732711953143734718e-02 3.37344131148970660660779685713351e-01 3.73872055611910916045337316404584e-03
            5.36987667512220934185052101383917e-02 3.53970493619263082418768817660748e-01 2.97019122244824304926447133823331e-03
            2.20880252143010324950012090994278e-02 3.65794709199201262261880174264661e-01 1.99663497484075559695870794030270e-03
            4.21741578953443996269356830453034e-03 3.72479333732178785609789883892518e-01 8.84006659974903292835424206685957e-04
            9.95782584210465393503852737921989e-01 1.24044496991100139209052155564450e-03 3.52540747732509089518312082878193e-06
            9.77911974785698800971545097127091e-01 6.49662758894605212728334819871634e-03 4.24646638563475371385032286486449e-05
            9.46301233248777906581494789861608e-01 1.57941185861420717306824457182302e-02 1.58705218426027433930788901328413e-04
            9.01852479486261682239955916884355e-01 2.88675824737542330300499315853813e-02 3.83123442062470204606122736734619e-04
            8.45843521530176500888842383574229e-01 4.53411847064260076223263240535744e-02 7.30601981623605744448068932683782e-04
            7.79885415536973658490182970126625e-01 6.47410743277321704125526480311237e-02 1.19716490099947264064139673678255e-03
            7.05875580731421203850572965166066e-01 8.65091740100787376377411419525743e-02 1.75940438664715982121367066071116e-03
            6.25943112845752769324292330566095e-01 1.10019264707655314849965577650437e-01 2.37629498818122709968836403504611e-03
            5.42387506520867757053849800286116e-01 1.34595008894596107618113478565647e-01 2.99321295253737176475516967855128e-03
            4.57612493479132464990755124745192e-01 1.59529410373982716286889171897201e-01 3.54772068714671916581093746856368e-03
            3.74056887154247230675707669433905e-01 1.84105154560923578443976111884695e-01 3.97646864159612988681713474647950e-03
            2.94124419268578907171729497349588e-01 2.07615245258500114022837124139187e-01 4.22243279308244338909883452970462e-03
            2.20114584463026341509817029873375e-01 2.29383344940846722881389041504008e-01 4.24166099016072724314785347132783e-03
            1.54156478469823443600006385167944e-01 2.48783234562152871793827557667100e-01 4.00875110217573105142463418815169e-03
            9.81475205137383732711953143734718e-02 2.65256836794824701897255181393120e-01 3.52042338272806187154939649985863e-03
            5.36987667512220934185052101383917e-02 2.78330300682436859727175715306657e-01 2.79676709551526054761683148797147e-03
            2.20880252143010324950012090994278e-02 2.87627791679632927035470402188366e-01 1.88005504736046369189395033316714e-03
            4.21741578953443996269356830453034e-03 2.92883974298667915103777659169282e-01 8.32391099989939864216281772968387e-04
            9.95782584210465393503852737921989e-01 9.28314724021216098291675589138094e-04 3.20538270910145263667954430852891e-06
            9.77911974785698800971545097127091e-01 4.86189649165474479380888439550290e-03 3.86098628735591397394960599953606e-05
            9.46301233248777906581494789861608e-01 1.18198817296222263156790788229955e-02 1.44298486418637444484228127272729e-04
            9.01852479486261682239955916884355e-01 2.16037006939578772601873168923703e-02 3.48344769941392594803791782354097e-04
            8.45843521530176500888842383574229e-01 3.39320892006686539166260274669185e-02 6.64280363105275071210209514305234e-04
            7.79885415536973658490182970126625e-01 4.84504302933307443024268934550491e-02 1.08849025205973474680898327449086e-03
            7.05875580731421203850572965166066e-01 6.47410743277321426569770324022102e-02 1.59969150674040146674348417121791e-03
            6.25943112845752769324292330566095e-01 8.23353762814902634392666413987172e-02 2.16058283073135987129020918473543e-03
            5.42387506520867757053849800286116e-01 1.00727183847248533132834324987925e-01 2.72149903363839290593162445475173e-03
            4.57612493479132464990755124745192e-01 1.19387400615777752865831473627622e-01 3.22567039993061849614819358578188e-03
            3.74056887154247230675707669433905e-01 1.37779208181536078070550388474658e-01 3.61549804073357881201156160955179e-03
            2.94124419268578907171729497349588e-01 1.55373510135294157219476574027794e-01 3.83913438441979820020533153979159e-03
            2.20114584463026341509817029873375e-01 1.71664154169695576390708424696641e-01 3.85661710970427833167217812615490e-03
            1.54156478469823443600006385167944e-01 1.86182495262357666776509290684771e-01 3.64485000688633074134026301749145e-03
            9.81475205137383732711953143734718e-02 1.98510883769068474657970568841847e-01 3.20085105403899615114649002123315e-03
            5.36987667512220934185052101383917e-02 2.08294702733404113459414475073572e-01 2.54288587830150524665695499493268e-03
            2.20880252143010324950012090994278e-02 2.15252687971371620134775071164768e-01 1.70938990165772435568525278171137e-03
            4.21741578953443996269356830453034e-03 2.19186269739005151757638145681995e-01 7.56829403772111327514116041470515e-04
            9.95782584210465393503852737921989e-01 6.50141966357684975355568646193660e-04 2.79314703782444791784942382528723e-06
            9.77911974785698800971545097127091e-01 3.40501218338933172788407510722664e-03 3.36443519863875652349538558638642e-05
            9.46301233248777906581494789861608e-01 8.27801278054083981838839889633164e-03 1.25740645183598819586284767169104e-04
            9.01852479486261682239955916884355e-01 1.51300761329426648044149672500680e-02 3.03545083568566788530251887578970e-04
            8.45843521530176500888842383574229e-01 2.37642198542171376174358954358468e-02 5.78849047642292550107379067725333e-04
            7.79885415536973658490182970126625e-01 3.39320892006686469777321235596901e-02 9.48502440787705211437674801544517e-04
            7.05875580731421203850572965166066e-01 4.53411847064259798667507084246608e-02 1.39395947348122379964852068923165e-03
            6.25943112845752769324292330566095e-01 5.76632924710828928538397519787395e-02 1.88271606897242129753655692780967e-03
            5.42387506520867757053849800286116e-01 7.05439304985380627455171520523436e-02 2.37149434376903887541909732306067e-03
            4.57612493479132464990755124745192e-01 8.36125479712853392211258096722304e-02 2.81082558316097857672999360545418e-03
            3.74056887154247230675707669433905e-01 9.64931859987405576850605370964331e-02 3.15051853685389742984468419706445e-03
            2.94124419268578907171729497349588e-01 1.08815293763397435977680061114370e-01 3.34539361031810714153245100987988e-03
            2.20114584463026341509817029873375e-01 1.20224389269154782744486453793797e-01 3.36062792920389539694236979983089e-03
            1.54156478469823443600006385167944e-01 1.30392258615606299043676585824869e-01 3.17609562538098428446931009716536e-03
            9.81475205137383732711953143734718e-02 1.39026402336880794408102701709140e-01 2.78919818676270003451889145651421e-03
            5.36987667512220934185052101383917e-02 1.45878465689282615924682318109262e-01 2.21585214718229882649436213171157e-03
            2.20880252143010324950012090994278e-02 1.50751466286434127050952724857780e-01 1.48954985211133413523787449861402e-03
            4.21741578953443996269356830453034e-03 1.53506336503465762310938202972466e-01 6.59495604466245896949083693527882e-04
            9.95782584210465393503852737921989e-01 4.13928902718311924301131110581764e-04 2.30056297935330941479420419071289e-06
            9.77911974785698800971545097127091e-01 2.16788490782859194494491816840309e-03 2.77110190033181988345997726064240e-05
            9.46301233248777906581494789861608e-01 5.27040081127802279709060329082604e-03 1.03565716158895819182933495383736e-04
            9.01852479486261682239955916884355e-01 9.63293578299469557824075138796616e-03 2.50013541129746148447554610072530e-04
            8.45843521530176500888842383574229e-01 1.51300761329426648044149672500680e-02 4.76766411365514238863971829829325e-04
            7.79885415536973658490182970126625e-01 2.16037006939578703212934129851419e-02 7.81229763973347837886784805760954e-04
            7.05875580731421203850572965166066e-01 2.88675824737542191522621237709245e-02 1.14812844292921586997202609836677e-03
            6.25943112845752769324292330566095e-01 3.67127560052765977705035993494675e-02 1.55069061179290883581427529236407e-03
            5.42387506520867757053849800286116e-01 4.49135315910860963284356728308921e-02 1.95327063671888291954137351069676e-03
            4.57612493479132464990755124745192e-01 5.32339889226522491871840259136661e-02 2.31512383360812836613495946380681e-03
            3.74056887154247230675707669433905e-01 6.14347645084617755006917150240042e-02 2.59491040518145931609295118391856e-03
            2.94124419268578907171729497349588e-01 6.92799380399841402411453827880905e-02 2.75541838186130899623904610962200e-03
            2.20114584463026341509817029873375e-01 7.65438198197804925415610455274873e-02 2.76796606000703467390100520617580e-03
            1.54156478469823443600006385167944e-01 8.30174443807957102015038231002109e-02 2.61597685896575041578082654325499e-03
            9.81475205137383732711953143734718e-02 8.85145847307436811624015149391198e-02 2.29731052595912163191704813414162e-03
            5.36987667512220934185052101383917e-02 9.28771197024603478720194971174351e-02 1.82507664240214376172311006030213e-03
            2.20880252143010324950012090994278e-02 9.59796356059098021429321079267538e-02 1.22686102781672102916832312047291e-03
            4.21741578953443996269356830453034e-03 9.77335916110200764173043808114016e-02 5.43190584718739933635378580589759e-04
            9.95782584210465393503852737921989e-01 2.26470026775140012348028206723427e-04 1.74181340640683852540667071739922e-06
            9.77911974785698800971545097127091e-01 1.18609971397786958559328329698701e-03 2.09807011754758989589917522744500e-05
            9.46301233248777906581494789861608e-01 2.88355755060215558291392135004116e-03 7.84121775707259937053181242383459e-05
            9.01852479486261682239955916884355e-01 5.27040081127802279709060329082604e-03 1.89291465450536146482296562254533e-04
            8.45843521530176500888842383574229e-01 8.27801278054083981838839889633164e-03 3.60971698881448342828559550099499e-04
            7.79885415536973658490182970126625e-01 1.18198817296222228462321268693813e-02 5.91488469815913318027023048983892e-04
            7.05875580731421203850572965166066e-01 1.57941185861420613223415898573876e-02 8.69276578002341942033592836480693e-04
            6.25943112845752769324292330566095e-01 2.00863935349841274402571400514717e-02 1.17406640072482124949326820484430e-03
            5.42387506520867757053849800286116e-01 2.45732265497810610510587281396511e-02 1.47886974271582141357561646088925e-03
            4.57612493479132464990755124745192e-01 2.91255402014410184896586741842839e-02 1.75283779103677355565293893846501e-03
            3.74056887154247230675707669433905e-01 3.36123732162379659782480700869201e-02 1.96467115777034786669452692819959e-03
            2.94124419268578907171729497349588e-01 3.79046481650800182183758124665474e-02 2.08619573593894378979451964539749e-03
            2.20114584463026341509817029873375e-01 4.18788850215998653681026553385891e-02 2.09569589490422739444519528717592e-03
            1.54156478469823443600006385167944e-01 4.54207539706812518653933352652530e-02 1.98062109348445033324415121001039e-03
            9.81475205137383732711953143734718e-02 4.84283659399440749582232967895834e-02 1.73935089311050544019776609161454e-03
            5.36987667512220934185052101383917e-02 5.08152092006199374019104197941488e-02 1.38181088367754132360809560253756e-03
            2.20880252143010324950012090994278e-02 5.25126670372442344580932171993481e-02 9.28886974722135548218626688310451e-04
            4.21741578953443996269356830453034e-03 5.34722967244469563063979933303926e-02 4.11263091333857343148000529353681e-04
            9.95782584210465393503852737921989e-01 9.31543862984316882631941059145220e-05 1.13304076349583715411085786672896e-06
            9.77911974785698800971545097127091e-01 4.87880857867600630471682787714371e-04 1.36478394247625573772577170506004e-05
            9.46301233248777906581494789861608e-01 1.18609971397786351406111737816218e-03 5.10067227725511117318937803766232e-05
            9.01852479486261682239955916884355e-01 2.16788490782858110292319331335875e-03 1.23133135701233685380465154857177e-04
            8.45843521530176500888842383574229e-01 3.40501218338931438064931533915569e-03 2.34810254529351002204112797144830e-04
            7.79885415536973658490182970126625e-01 4.86189649165471877295674474339648e-03 3.84760241811269788122368273874940e-04
            7.05875580731421203850572965166066e-01 6.49662758894601569809035268576736e-03 5.65459993594038894636211178834628e-04
            6.25943112845752769324292330566095e-01 8.26217795504596841382749516924378e-03 7.63724223374905183091276850149143e-04
            5.42387506520867757053849800286116e-01 1.01077562943462390815385631981371e-02 9.61997247371196902848700105437274e-04
            4.57612493479132464990755124745192e-01 1.19802689199547882092922179708694e-02 1.14021206963702752094913517311170e-03
            3.74056887154247230675707669433905e-01 1.38258472592550640811737139301840e-02 1.27800859749406590763765034779453e-03
            2.94124419268578907171729497349588e-01 1.55913976253550124601021664716427e-02 1.35705971762277053868639420386444e-03
            2.20114584463026341509817029873375e-01 1.72261287226463111199592503908207e-02 1.36323952272001205389406752743753e-03
            1.54156478469823443600006385167944e-01 1.86830130309117159459475487892632e-02 1.28838394956837134879001904863571e-03
            9.81475205137383732711953143734718e-02 1.99201403064724544278440987454815e-02 1.13143891111880693853275481330911e-03
            5.36987667512220934185052101383917e-02 2.09019255003231700651422642067701e-02 8.98860953125060237189902778709438e-04
            2.20880252143010324950012090994278e-02 2.16001443564334365265988679993825e-02 6.04236253532819704223366663597972e-04
            4.21741578953443996269356830453034e-03 2.19948708280026036798737720801000e-02 2.67524549580675668515594711749372e-04
            9.95782584210465393503852737921989e-01 1.77865959418151070917362560619424e-05 4.92649033616421286018165377751377e-07
            9.77911974785698800971545097127091e-01 9.31543862984284763142581176076362e-05 5.93411562953539969354202140983112e-06
            9.46301233248777906581494789861608e-01 2.26470026775131067680105201311846e-04 2.21778540467580794967594748356987e-05
            9.01852479486261682239955916884355e-01 4.13928902718295552848326579464810e-04 5.35386036087617838601440678480969e-05
            8.45843521530176500888842383574229e-01 6.50141966357659279764080739738574e-04 1.02096101662043766355146556801259e-04
            7.79885415536973658490182970126625e-01 9.28314724021179126997593833436895e-04 1.67294741203756426508458821089675e-04
            7.05875580731421203850572965166066e-01 1.24044496991095173563102171954142e-03 2.45863457315827216038278679732798e-04
            6.25943112845752769324292330566095e-01 1.57755342206842446607650831680303e-03 3.32069253567046369490095569787513e-04
            5.42387506520867757053849800286116e-01 1.92994215548711800164993768191835e-03 4.18278873565715921412772049237105e-04
            4.57612493479132464990755124745192e-01 2.28747363404732066000102364000668e-03 4.95767135942523553410188341672438e-04
            3.74056887154247230675707669433905e-01 2.63986236746601527977662549062643e-03 5.55681420116217023551230536071444e-04
            2.94124419268578907171729497349588e-01 2.97697081962348779338167759078715e-03 5.90053050151436819618921969521352e-04
            2.20114584463026341509817029873375e-01 3.28910106551326061885553997399256e-03 5.92740045277455633240815124906931e-04
            1.54156478469823443600006385167944e-01 3.56727382317678035766883581914044e-03 5.60192649842026807634198792840152e-04
            9.81475205137383732711953143734718e-02 3.80348688681614484352611071926731e-03 4.91952544089332001291114959684592e-04
            5.36987667512220934185052101383917e-02 3.99094576275930892211851741535611e-03 3.90827050693506037378627260281405e-04
            2.20880252143010324950012090994278e-02 4.12426140323601201492786927360612e-03 2.62723474714636115717042441630724e-04
            4.21741578953443996269356830453034e-03 4.19962919359262529944709640972178e-03 1.16320361160662233131028731047962e-04
            ];
        
    case  35
        
        % ALG. DEG.:   35
        % PTS CARD.:  361
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.8e-15
        
        xyw=[
            9.96203421921792342530466157768387e-01 3.78216407310378579728626924350010e-03 3.59499118241727625512388008544029e-07
            9.80104076067414897721619126969017e-01 1.98203875039369079025242115221772e-02 4.33813207936495602985672512263982e-06
            9.51577951807409005979820904030930e-01 4.82382101059210879889782574991841e-02 1.62665429629977470200180578174809e-05
            9.11357328268571409424225748807658e-01 8.83061329071393452672111834544921e-02 3.94583861731775495276623721974119e-05
            8.60483088667614692823804034560453e-01 1.38987224485281529728908367360418e-01 7.57327940529919240884512698919195e-05
            8.00272652330840550405355315888301e-01 1.98969067199380095223304465434921e-01 1.25118315784170422953758272655023e-04
            7.32285370687980496917646178189898e-01 2.66698229819158028064407517376821e-01 1.85752553207852810854966429943147e-04
            6.58282049981815053918410285405116e-01 3.40420591140215833547699730843306e-01 2.53990440001341459107936371353276e-04
            5.80179322820112863823283078090753e-01 4.18226795200127676732648751567467e-01 3.24712188311432803280304204562867e-04
            5.00000000000000111022302462515654e-01 4.98101710960896060242930616368540e-01 3.91800949551500778990364981524408e-04
            4.19820677179887136176716921909247e-01 5.77976626721664610286666174943093e-01 4.48742303002960200158932835279302e-04
            3.41717950018184946081589714594884e-01 6.55782830781576508982766426925082e-01 4.89284649843442281599847909490109e-04
            2.67714629312019669615807515583583e-01 7.29505192102634203443756177875912e-01 5.08092806252723573970375703368063e-04
            1.99727347669159671639249609143008e-01 7.97234354722411997506981151673244e-01 5.01327272385479168551047912671947e-04
            1.39516911332385140642742271666066e-01 8.57216197436511007090587099810364e-01 4.67088813232778153790303310444187e-04
            8.86426717314284795534717886766884e-02 9.07897289014653163796708668087376e-01 4.05681470314105498738838084804570e-04
            4.84220481925909940201790959690697e-02 9.47965211815871233724806188547518e-01 3.19666024331556651930097423530697e-04
            1.98959239325848247226247167418478e-02 9.76383034417855566466926120483549e-01 2.13703115668870732152262448266811e-04
            3.79657807820771298068507348943967e-03 9.92421257848688620484267630672548e-01 9.43308011564378891570734064764281e-05
            9.96203421921792342530466157768387e-01 3.72104164955957238125616726165390e-03 8.27810621346032892518677394344540e-07
            9.80104076067414897721619126969017e-01 1.95000761434538337490973702870178e-02 9.98932022327122291305376983183706e-06
            9.51577951807409005979820904030930e-01 4.74586468050912368976668176401290e-02 3.74566065786475962387166993572407e-05
            9.11357328268571409424225748807658e-01 8.68790438774790368947620322614966e-02 9.08599479606135356987706153120143e-05
            8.60483088667614692823804034560453e-01 1.36741093477206948625379823170078e-01 1.74388219943070766357676171232072e-04
            8.00272652330840550405355315888301e-01 1.95753587552676827643338697271247e-01 2.88107162091618350906185863635756e-04
            7.32285370687980496917646178189898e-01 2.62388199411587363929498906145454e-01 4.27728271600963827914904102911464e-04
            6.58282049981815053918410285405116e-01 3.34919155678224222949523891657009e-01 5.84858135346208040002136119284160e-04
            5.80179322820112863823283078090753e-01 4.11467956921389710345238199806772e-01 7.47707531744140922388974779266846e-04
            5.00000000000000111022302462515654e-01 4.90052038033707337838507100968855e-01 9.02191329643565369592050906533132e-04
            4.19820677179887136176716921909247e-01 5.68636119146025187376380927162245e-01 1.03330891739030904415141964847180e-03
            3.41717950018184946081589714594884e-01 6.45184920389190730283246466569835e-01 1.12666487746328788759064831737078e-03
            2.67714629312019669615807515583583e-01 7.17715876655827478280968989565736e-01 1.16997400077822064426436021733480e-03
            1.99727347669159671639249609143008e-01 7.84350488514737764766948657779722e-01 1.15439515646346101129804395668543e-03
            1.39516911332385140642742271666066e-01 8.43362982590208143385268613201333e-01 1.07555501831861481144836378831542e-03
            8.86426717314284795534717886766884e-02 8.93225032189935985726947365037631e-01 9.34153695986205800244617236671729e-04
            4.84220481925909940201790959690697e-02 9.32645429262323633068376693699975e-01 7.36087842216039166601260212985380e-04
            1.98959239325848247226247167418478e-02 9.60603999923961171525377267244039e-01 4.92089409928620261612530750738870e-04
            3.79657807820771298068507348943967e-03 9.76383034417855344422321195452241e-01 2.17213437126912366995773462363672e-04
            9.96203421921792342530466157768387e-01 3.61273999153780478107989004854517e-03 1.27539422074654381310514911218590e-06
            9.80104076067414897721619126969017e-01 1.89325225450852896447173634442152e-02 1.53903815116925311900011874577388e-05
            9.51577951807409005979820904030930e-01 4.60773534414253907454117609177047e-02 5.77087782245487581852562419815200e-05
            9.11357328268571409424225748807658e-01 8.43504120089293846218581052198715e-02 1.39986428705000031866290122195551e-04
            8.60483088667614692823804034560453e-01 1.32761216728167097400614693469834e-01 2.68677064713208739194644225634079e-04
            8.00272652330840550405355315888301e-01 1.90056140414944979566413962857041e-01 4.43881970117583190410526228220078e-04
            7.32285370687980496917646178189898e-01 2.54751338629611245867323532365845e-01 6.58993919119749680703146399451953e-04
            6.58282049981815053918410285405116e-01 3.25171266974131012439386267942609e-01 9.01081318048648047432103158627115e-04
            5.80179322820112863823283078090753e-01 3.99492100117236459411174109845888e-01 1.15198070694543202967252693014188e-03
            5.00000000000000111022302462515654e-01 4.75788975903704391967607989499811e-01 1.38999135570895332100760377613824e-03
            4.19820677179887136176716921909247e-01 5.52085851690172546568646794185042e-01 1.59200206847138646046313503745750e-03
            3.41717950018184946081589714594884e-01 6.26406684833277993540434636088321e-01 1.73583406202049064098735797756490e-03
            2.67714629312019669615807515583583e-01 6.96826613177797593579043677891605e-01 1.80255971660517018764768870653370e-03
            1.99727347669159671639249609143008e-01 7.61521811392463776613226400513668e-01 1.77855764717937936923719544068945e-03
            1.39516911332385140642742271666066e-01 8.18816735079242130623811135592405e-01 1.65708994193382893982124937082290e-03
            8.86426717314284795534717886766884e-02 8.67227539798479774013628684770083e-01 1.43923524828972696637530148677797e-03
            4.84220481925909940201790959690697e-02 9.05500598365983622173303047020454e-01 1.13407844223794001398131214841669e-03
            1.98959239325848247226247167418478e-02 9.32645429262323855112981618731283e-01 7.58154067282972195240819957007261e-04
            3.79657807820771298068507348943967e-03 9.47965211815871233724806188547518e-01 3.34657173073833237326307754599952e-04
            9.96203421921792342530466157768387e-01 3.46003925391840869152382254014810e-03 1.69000822087614873080665001819911e-06
            9.80104076067414897721619126969017e-01 1.81322960786354375750661915844830e-02 2.03935934898277640772712437922110e-05
            9.51577951807409005979820904030930e-01 4.41297884700917325551117187387717e-02 7.64691481502231573978353718068490e-05
            9.11357328268571409424225748807658e-01 8.07851484797428298634258680976927e-02 1.85494187972769263624689406455559e-04
            8.60483088667614692823804034560453e-01 1.27149759560165859229030616006639e-01 3.56020468604923497231429241338674e-04
            8.00272652330840550405355315888301e-01 1.82022981953933182808214041870087e-01 5.88182199977598202216000355946335e-04
            7.32285370687980496917646178189898e-01 2.43983689308213069768527248015744e-01 8.73224233498461803806844905295748e-04
            6.58282049981815053918410285405116e-01 3.11427157949986266327613293469767e-01 1.19401108332508297527052576469941e-03
            5.80179322820112863823283078090753e-01 3.82606650706564355157723866796005e-01 1.52647458594325257166812725984073e-03
            5.00000000000000111022302462515654e-01 4.55678664134285593689810411888175e-01 1.84185938738211114201748230811972e-03
            4.19820677179887136176716921909247e-01 5.28750677562007109777653113269480e-01 2.10954114390890946212353540545337e-03
            3.41717950018184946081589714594884e-01 5.99930170318585198607763686595717e-01 2.30013103962029789167909044067528e-03
            2.67714629312019669615807515583583e-01 6.67373638960358284144547269534087e-01 2.38854833284391290804116181334393e-03
            1.99727347669159671639249609143008e-01 7.29334346314638004571406781906262e-01 2.35674350419737624329208358631149e-03
            1.39516911332385140642742271666066e-01 7.84207568708405799995375673461240e-01 2.19578823476250257346054084450770e-03
            8.86426717314284795534717886766884e-02 8.30572179788828690583102343225619e-01 1.90711182614628849477100835940746e-03
            4.84220481925909940201790959690697e-02 8.67227539798479662991326222254429e-01 1.50275252884450485728329294232708e-03
            1.98959239325848247226247167418478e-02 8.93225032189936096749249827553285e-01 1.00462005045696294892876299087447e-03
            3.79657807820771298068507348943967e-03 9.07897289014653052774406205571722e-01 4.43449848266440761583584428962013e-04
            9.96203421921792342530466157768387e-01 3.26689123110392946186530949148619e-03 2.06086461459863778353804666942928e-06
            9.80104076067414897721619126969017e-01 1.71201060774066951553074034109159e-02 2.48687755885036074459792238000588e-05
            9.51577951807409005979820904030930e-01 4.16663535883727911834917279065849e-02 9.32495828035648764448711789398772e-05
            9.11357328268571409424225748807658e-01 7.62755199592091842619723252028052e-02 2.26199141214002060482415878972517e-04
            8.60483088667614692823804034560453e-01 1.20051942784656640039742114822729e-01 4.34145808734785775863385115513893e-04
            8.00272652330840550405355315888301e-01 1.71862005013748792014993682641943e-01 7.17253246402666118884672297184579e-04
            7.32285370687980496917646178189898e-01 2.30363911111912067486429123164271e-01 1.06484507069086074479424830485641e-03
            6.58282049981815053918410285405116e-01 2.94042517084813381789132336052717e-01 1.45602557470847927129087295838872e-03
            5.80179322820112863823283078090753e-01 3.61248592986278871386218725092476e-01 1.86144506304451794014009902156204e-03
            5.00000000000000111022302462515654e-01 4.30241544333807235389599554764573e-01 2.24603809001251751922834110075655e-03
            4.19820677179887136176716921909247e-01 4.99234495681335821437585309467977e-01 2.57246008795622730855323645471344e-03
            3.41717950018184946081589714594884e-01 5.66440571582801366545822929765563e-01 2.80487314199925202862218043264875e-03
            2.67714629312019669615807515583583e-01 6.30119177555702569826223680138355e-01 2.91269277782840426208044171119127e-03
            1.99727347669159671639249609143008e-01 6.88621083653865651008629811258288e-01 2.87390867895754827457399649404124e-03
            1.39516911332385140642742271666066e-01 7.40431145882958219317515613511205e-01 2.67763329093632625535481572853769e-03
            8.86426717314284795534717886766884e-02 7.84207568708405688973073210945586e-01 2.32560956215342859096373295813009e-03
            4.84220481925909940201790959690697e-02 8.18816735079241908579206210561097e-01 1.83251742384347732911509254449811e-03
            1.98959239325848247226247167418478e-02 8.43362982590208143385268613201333e-01 1.22507446267314918340873841628991e-03
            3.79657807820771298068507348943967e-03 8.57216197436510785045982174779056e-01 5.40760742671213745155767149697112e-04
            9.96203421921792342530466157768387e-01 3.03829760842841195209174109947980e-03 2.37834958723482436934791410532153e-06
            9.80104076067414897721619126969017e-01 1.59221638161024837965218381441446e-02 2.86999164025493689049624290721141e-05
            9.51577951807409005979820904030930e-01 3.87508409383765795030996059722384e-02 1.07615078253876813424959324461128e-04
            9.11357328268571409424225748807658e-01 7.09383060162024209338582636519277e-02 2.61046082468628138934657600955802e-04
            8.60483088667614692823804034560453e-01 1.11651568676974691229908387413161e-01 5.01027819920718260442160651990662e-04
            8.00272652330840550405355315888301e-01 1.59836334262202117706408444064436e-01 8.27749164326762828548800232653093e-04
            7.32285370687980496917646178189898e-01 2.14244696467297629505210920797253e-01 1.22888898979900992695923900299704e-03
            6.58282049981815053918410285405116e-01 2.73467530210110443267268465206143e-01 1.68033251678964914467562241640053e-03
            5.80179322820112863823283078090753e-01 3.35971006830077856264438196376432e-01 2.14820860428740004199887891900289e-03
            5.00000000000000111022302462515654e-01 4.00136326165420164180375195428496e-01 2.59204982532795563318628317972525e-03
            4.19820677179887136176716921909247e-01 4.64301645500762694140917119511869e-01 2.96875852252929256502866905975679e-03
            3.41717950018184946081589714594884e-01 5.26805122120730051626935619424330e-01 3.23697579756793748367860175108035e-03
            2.67714629312019669615807515583583e-01 5.86027955863542837633417548204307e-01 3.36140550758070059608750845825398e-03
            1.99727347669159671639249609143008e-01 6.40436318068638210654341946792556e-01 3.31664655306842691070978723644203e-03
            1.39516911332385140642742271666066e-01 6.88621083653865984075537198805250e-01 3.09013410543947748473914138855889e-03
            8.86426717314284795534717886766884e-02 7.29334346314638226616011706937570e-01 2.68387962170633627770333085038601e-03
            4.84220481925909940201790959690697e-02 7.61521811392463998657831325544976e-01 2.11482453904307966113806926955476e-03
            1.98959239325848247226247167418478e-02 7.84350488514738208856158507842338e-01 1.41380240215248473463482969947336e-03
            3.79657807820771298068507348943967e-03 7.97234354722412219551586076704552e-01 6.24067238582460279486263665660317e-04
            9.96203421921792342530466157768387e-01 2.78017858534619589569447661858703e-03 2.63423808139417025888924343834940e-06
            9.80104076067414897721619126969017e-01 1.45694940321529041449277741548940e-02 3.17877628781744695093090802284763e-05
            9.51577951807409005979820904030930e-01 3.54587575101827506829366143392690e-02 1.19193468777719437145037884029364e-04
            9.11357328268571409424225748807658e-01 6.49117317276221988953111008413543e-02 2.89132234860866156687247974232946e-04
            8.60483088667614692823804034560453e-01 1.02166193132277885458059074608173e-01 5.54933795333068034631718301596948e-04
            8.00272652330840550405355315888301e-01 1.46257414824437548261926167469937e-01 9.16807344981985204712682246963595e-04
            7.32285370687980496917646178189898e-01 1.96043506564347491671895795661840e-01 1.36110611833908804260695468713038e-03
            6.58282049981815053918410285405116e-01 2.50235055699803365936162435900769e-01 1.86112080784495972571312982779546e-03
            5.80179322820112863823283078090753e-01 3.07428540211152623307100384408841e-01 2.37933604990836185602565144847631e-03
            5.00000000000000111022302462515654e-01 3.66142685343990192947671857837122e-01 2.87093049541494674953279186979671e-03
            4.19820677179887136176716921909247e-01 4.24856830476827873610545793781057e-01 3.28816957628273731639723109765328e-03
            3.41717950018184946081589714594884e-01 4.82050314988177130981483742289129e-01 3.58524455793673242284813618141470e-03
            2.67714629312019669615807515583583e-01 5.36241864123632949734599151270231e-01 3.72306175786886808798392678454547e-03
            1.99727347669159671639249609143008e-01 5.86027955863542726611115085688652e-01 3.67348715239772553517449971138831e-03
            1.39516911332385140642742271666066e-01 6.30119177555702791870828605169663e-01 3.42260405318617648082191173841693e-03
            8.86426717314284795534717886766884e-02 6.67373638960358395166849732049741e-01 2.97264033148149755050870801653673e-03
            4.84220481925909940201790959690697e-02 6.96826613177797704601346140407259e-01 2.34236016694719384467759937251685e-03
            1.98959239325848247226247167418478e-02 7.17715876655827700325573914597044e-01 1.56591450950095993285837714381614e-03
            3.79657807820771298068507348943967e-03 7.29505192102634314466058640391566e-01 6.91211121379232788786151076010356e-04
            9.96203421921792342530466157768387e-01 2.49921920023859307422031683643127e-03 2.82190191218257094781548914574909e-06
            9.80104076067414897721619126969017e-01 1.30971295926243392476751026265447e-02 3.40523316717264792636427062078042e-05
            9.51577951807409005979820904030930e-01 3.18753651485370395612939375951100e-02 1.27684843613492084515859126270243e-04
            9.11357328268571409424225748807658e-01 5.83518796632299385773734456961392e-02 3.09730093187204240986337699936826e-04
            8.60483088667614692823804034560453e-01 9.18414783990137290059507790829230e-02 5.94467428455186259333986598107913e-04
            8.00272652330840550405355315888301e-01 1.31476927861084946780323434722959e-01 9.82120947298144829351596030164728e-04
            7.32285370687980496917646178189898e-01 1.76231734993637922936926543115987e-01 1.45807168499806566810961250268974e-03
            6.58282049981815053918410285405116e-01 2.24946792653554189511666550060909e-01 1.99370755572742149042131387659538e-03
            5.80179322820112863823283078090753e-01 2.76360415998729902486985565701616e-01 2.54884059128315885317528355358263e-03
            5.00000000000000111022302462515654e-01 3.29141024990907471448053911444731e-01 3.07545635756164593976880183845424e-03
            4.19820677179887136176716921909247e-01 3.81921633983085151431424719703500e-01 3.52241966298731754708706276346675e-03
            3.41717950018184946081589714594884e-01 4.33335257328260836651168119715294e-01 3.84065834638959088809251340990158e-03
            2.67714629312019669615807515583583e-01 4.82050314988177075470332511031302e-01 3.98829368078363038357947445433638e-03
            1.99727347669159671639249609143008e-01 5.26805122120729940604633156908676e-01 3.93518736705944505160914559382945e-03
            1.39516911332385140642742271666066e-01 5.66440571582801477568125392281218e-01 3.66643128825252595862083104805151e-03
            8.86426717314284795534717886766884e-02 5.99930170318585198607763686595717e-01 3.18441202975816746445492100292540e-03
            4.84220481925909940201790959690697e-02 6.26406684833277993540434636088321e-01 2.50923053645564107447252233384916e-03
            1.98959239325848247226247167418478e-02 6.45184920389190841305548929085489e-01 1.67747068113771710846926499272058e-03
            3.79657807820771298068507348943967e-03 6.55782830781576508982766426925082e-01 7.40453187932655731810782651081126e-04
            9.96203421921792342530466157768387e-01 2.20269609844823662306301237379103e-03 2.93648036621551472048763375966018e-06
            9.80104076067414897721619126969017e-01 1.15432036740876695080082470212801e-02 3.54349677946616507134315610105801e-05
            9.51577951807409005979820904030930e-01 2.80934711299403137851893319520968e-02 1.32869266190872436107089926693448e-04
            9.11357328268571409424225748807658e-01 5.14286452581058303734629078007856e-02 3.22306148751595310936179927807643e-04
            8.60483088667614692823804034560453e-01 8.09448271387770423856977686227765e-02 6.18604751808375890979874167641128e-04
            8.00272652330840550405355315888301e-01 1.15877677319350141926790342949971e-01 1.02199827234937469684628030819340e-03
            7.32285370687980496917646178189898e-01 1.55322492343285017968312899938610e-01 1.51727416783953121537376063088232e-03
            6.58282049981815053918410285405116e-01 1.98257688837027712391858358387253e-01 2.07465860811618637085662619767845e-03
            5.80179322820112863823283078090753e-01 2.43571276192108138181069421079883e-01 2.65233186192755966692491398362108e-03
            5.00000000000000111022302462515654e-01 2.90089661410056376400490307787550e-01 3.20032995198883717449800379029057e-03
            4.19820677179887136176716921909247e-01 3.36608046628004753397789272639784e-01 3.66544143057532514559659908570666e-03
            3.41717950018184946081589714594884e-01 3.81921633983085151431424719703500e-01 3.99660164615428178946920922953723e-03
            2.67714629312019669615807515583583e-01 4.24856830476827762588243331265403e-01 4.15023145835156318655956297902776e-03
            1.99727347669159671639249609143008e-01 4.64301645500762583118614656996215e-01 4.09496885446731465757475021405298e-03
            1.39516911332385140642742271666066e-01 4.99234495681335932459887771983631e-01 3.81530039919234257791980091667483e-03
            8.86426717314284795534717886766884e-02 5.28750677562007109777653113269480e-01 3.31370958110055197770593693462615e-03
            4.84220481925909940201790959690697e-02 5.52085851690172546568646794185042e-01 2.61111350922593613357869379854037e-03
            1.98959239325848247226247167418478e-02 5.68636119146025298398683389677899e-01 1.74558148133973104962346489799074e-03
            3.79657807820771298068507348943967e-03 5.77976626721664610286666174943093e-01 7.70518010948233081185143600322363e-04
            9.96203421921792342530466157768387e-01 1.89828903910385692402340573892161e-03 2.97500579217638840850046651198557e-06
            9.80104076067414897721619126969017e-01 9.94796196629252511833829686338504e-03 3.58998601344522795682755178514611e-05
            9.51577951807409005979820904030930e-01 2.42110240962955039489834518917633e-02 1.34612456826847690051149331047498e-04
            9.11357328268571409424225748807658e-01 4.43213358657143299823566451323131e-02 3.26534674102325569479410294349009e-04
            8.60483088667614692823804034560453e-01 6.97584556661926674658857905342302e-02 6.26720594106873062252571759955799e-04
            8.00272652330840550405355315888301e-01 9.98636738345797247973223420558497e-02 1.03540647327812112069023164906412e-03
            7.32285370687980496917646178189898e-01 1.33857314656009779296752526533965e-01 1.53718018672117017961453733221333e-03
            6.58282049981815053918410285405116e-01 1.70858975009092500796370472926355e-01 2.10187728375271949687874695200662e-03
            5.80179322820112863823283078090753e-01 2.09910338589943623599509692212450e-01 2.68712937528605907977485856008570e-03
            5.00000000000000111022302462515654e-01 2.50000000000000000000000000000000e-01 3.24231697701179381182479133371999e-03
            4.19820677179887136176716921909247e-01 2.90089661410056487422792770303204e-01 3.71353052869163261645590878856638e-03
            3.41717950018184946081589714594884e-01 3.29141024990907582470356373960385e-01 4.04903543137058018891449862053378e-03
            2.67714629312019669615807515583583e-01 3.66142685343990248458823089094949e-01 4.20468080410873416147277126242443e-03
            1.99727347669159671639249609143008e-01 4.00136326165420275202677657944150e-01 4.14869317737780012034232157702718e-03
            1.39516911332385140642742271666066e-01 4.30241544333807568456506942311535e-01 3.86535558591814733947478188724745e-03
            8.86426717314284795534717886766884e-02 4.55678664134285871245566568177310e-01 3.35718410066189161880534364001960e-03
            4.84220481925909940201790959690697e-02 4.75788975903704614012212914531119e-01 2.64537025459105675645465893808250e-03
            1.98959239325848247226247167418478e-02 4.90052038033707670905414488515817e-01 1.76848279915262540824827386387597e-03
            3.79657807820771298068507348943967e-03 4.98101710960896282287535541399848e-01 7.80626893310883683031742297231403e-04
            9.96203421921792342530466157768387e-01 1.59388197975947657446249561274954e-03 2.93648036621552742598184257416527e-06
            9.80104076067414897721619126969017e-01 8.35272025849737725922139475187578e-03 3.54349677946618065674938558018425e-05
            9.51577951807409005979820904030930e-01 2.03285770626506802349897640169729e-02 1.32869266190873005313230481583275e-04
            9.11357328268571409424225748807658e-01 3.72140264733228157134625746493839e-02 3.22306148751596720399004158963407e-04
            8.60483088667614692823804034560453e-01 5.85720841936082717293921007239987e-02 6.18604751808378601485305381402213e-04
            8.00272652330840550405355315888301e-01 8.38496703498092521567031099039013e-02 1.02199827234937925049540474731202e-03
            7.32285370687980496917646178189898e-01 1.12392136968734485114040921871492e-01 1.51727416783953793742723004100981e-03
            6.58282049981815053918410285405116e-01 1.43460261181157233689731356207631e-01 2.07465860811619504447400608171392e-03
            5.80179322820112863823283078090753e-01 1.76249400987778997995647500829364e-01 2.65233186192757137630837682706897e-03
            5.00000000000000111022302462515654e-01 2.09910338589943512577207229696796e-01 3.20032995198885105228581160474732e-03
            4.19820677179887136176716921909247e-01 2.43571276192108138181069421079883e-01 3.66544143057534075810788287697051e-03
            3.41717950018184946081589714594884e-01 2.76360415998729902486985565701616e-01 3.99660164615429913670396899760817e-03
            2.67714629312019669615807515583583e-01 3.07428540211152623307100384408841e-01 4.15023145835158140115606073550225e-03
            1.99727347669159671639249609143008e-01 3.35971006830077745242135733860778e-01 4.09496885446733200480950998212393e-03
            1.39516911332385140642742271666066e-01 3.61248592986278982408521187608130e-01 3.81530039919235949147369169054400e-03
            8.86426717314284795534717886766884e-02 3.82606650706564410668875098053832e-01 3.31370958110056628917461374328468e-03
            4.84220481925909940201790959690697e-02 3.99492100117236459411174109845888e-01 2.61111350922594784296215664198826e-03
            1.98959239325848247226247167418478e-02 4.11467956921389821367540662322426e-01 1.74558148133973863903867229652178e-03
            3.79657807820771298068507348943967e-03 4.18226795200127676732648751567467e-01 7.70518010948236442211878305386108e-04
            9.96203421921792342530466157768387e-01 1.29735887796911990646475665300841e-03 2.82190191218257560649669904440096e-06
            9.80104076067414897721619126969017e-01 6.79879433996070665219280115820766e-03 3.40523316717265334737513304830259e-05
            9.51577951807409005979820904030930e-01 1.65466830440539509894382064203455e-02 1.27684843613492301356293623371130e-04
            9.11357328268571409424225748807658e-01 3.02907920681987109789989887076445e-02 3.09730093187204728877315318413821e-04
            8.60483088667614692823804034560453e-01 4.76754329333715851091390902638523e-02 5.94467428455187235115941835061903e-04
            8.00272652330840550405355315888301e-01 6.82504198080744611809578259453701e-02 9.82120947298146347234637509870936e-04
            7.32285370687980496917646178189898e-01 9.14828943183815940232150865085714e-02 1.45807168499806805335439197079950e-03
            6.58282049981815053918410285405116e-01 1.16771157364630742692135356719518e-01 1.99370755572742452618739683600779e-03
            5.80179322820112863823283078090753e-01 1.43460261181157233689731356207631e-01 2.54884059128316318998397349560037e-03
            5.00000000000000111022302462515654e-01 1.70858975009092445285219241668528e-01 3.07545635756165114393922976887552e-03
            4.19820677179887136176716921909247e-01 1.98257688837027712391858358387253e-01 3.52241966298732318493835968808980e-03
            3.41717950018184946081589714594884e-01 2.24946792653554189511666550060909e-01 3.84065834638959695962467932872642e-03
            2.67714629312019669615807515583583e-01 2.50235055699803310425011204642942e-01 3.98829368078363645511164037316121e-03
            1.99727347669159671639249609143008e-01 2.73467530210110387756117233948316e-01 3.93518736705945199050304950105783e-03
            1.39516911332385140642742271666066e-01 2.94042517084813437300283567310544e-01 3.66643128825253203015299696687634e-03
            8.86426717314284795534717886766884e-02 3.11427157949986266327613293469767e-01 3.18441202975817266862534893334669e-03
            4.84220481925909940201790959690697e-02 3.25171266974131012439386267942609e-01 2.50923053645564497760034328166512e-03
            1.98959239325848247226247167418478e-02 3.34919155678224278460675122914836e-01 1.67747068113771971055447895793122e-03
            3.79657807820771298068507348943967e-03 3.40420591140215833547699730843306e-01 7.40453187932656924433172385136004e-04
            9.96203421921792342530466157768387e-01 1.01639949286151795235233485925619e-03 2.63423808139417703515282147275212e-06
            9.80104076067414897721619126969017e-01 5.32642990043214609174881957187608e-03 3.17877628781745508244720166413089e-05
            9.51577951807409005979820904030930e-01 1.29632906824082502761363855370291e-02 1.19193468777719748853162473611889e-04
            9.11357328268571409424225748807658e-01 2.37309400038064645388491413768861e-02 2.89132234860866915628768714086050e-04
            8.60483088667614692823804034560453e-01 3.73507182001074494737125064602878e-02 5.54933795333069444094542532752712e-04
            8.00272652330840550405355315888301e-01 5.34699328447218805160368049200770e-02 9.16807344981987481537244466522907e-04
            7.32285370687980496917646178189898e-01 7.16711227476720530438214495916327e-02 1.36110611833909151205390664074457e-03
            6.58282049981815053918410285405116e-01 9.14828943183816495343663177663984e-02 1.86112080784496449620268876401497e-03
            5.80179322820112863823283078090753e-01 1.12392136968734554502979960943776e-01 2.37933604990836792755781736730114e-03
            5.00000000000000111022302462515654e-01 1.33857314656009807052328142162878e-01 2.87093049541495412210756477122686e-03
            4.19820677179887136176716921909247e-01 1.55322492343285101235039746825350e-01 3.28816957628274599001461098168875e-03
            3.41717950018184946081589714594884e-01 1.76231734993638033959229005631641e-01 3.58524455793674153014638505965195e-03
            2.67714629312019669615807515583583e-01 1.96043506564347574938622642548580e-01 3.72306175786887762896304465698449e-03
            1.99727347669159671639249609143008e-01 2.14244696467297712771937767683994e-01 3.67348715239773464247274858962555e-03
            1.39516911332385140642742271666066e-01 2.30363911111912289531034048195579e-01 3.42260405318618515443929162245240e-03
            8.86426717314284795534717886766884e-02 2.43983689308213236301980941789225e-01 2.97264033148150535676434991216865e-03
            4.84220481925909940201790959690697e-02 2.54751338629611412400777226139326e-01 2.34236016694719948252889629713991e-03
            1.98959239325848247226247167418478e-02 2.62388199411587585974103831176762e-01 1.56591450950096383598619809163210e-03
            3.79657807820771298068507348943967e-03 2.66698229819158139086709979892476e-01 6.91211121379234523509627052817450e-04
            9.96203421921792342530466157768387e-01 7.58280469779302004375287626913860e-04 2.37834958723483029857854488542390e-06
            9.80104076067414897721619126969017e-01 3.97376011648256817487823155943261e-03 2.86999164025494434438617874505439e-05
            9.51577951807409005979820904030930e-01 9.67120725421442666014382183448106e-03 1.07615078253877084475502445837236e-04
            9.11357328268571409424225748807658e-01 1.77043657152262425003019785663128e-02 2.61046082468628789455961092258462e-04
            8.60483088667614692823804034560453e-01 2.78653426554106437018631936552993e-02 5.01027819920719561484767634595983e-04
            8.00272652330840550405355315888301e-01 3.98910134069573249493423361400346e-02 8.27749164326764996953145203661961e-04
            7.32285370687980496917646178189898e-01 5.34699328447219221494002283634472e-02 1.22888898979901317956575645951034e-03
            6.58282049981815053918410285405116e-01 6.82504198080745583254724806465674e-02 1.68033251678965348148431235841826e-03
            5.80179322820112863823283078090753e-01 8.38496703498093631790055724195554e-02 2.14820860428740567985017584362595e-03
            5.00000000000000111022302462515654e-01 9.98636738345798080640491889425903e-02 2.59204982532796213839931809275186e-03
            4.19820677179887136176716921909247e-01 1.15877677319350308460244036723452e-01 2.96875852252930037128431095538872e-03
            3.41717950018184946081589714594884e-01 1.31476927861085113313777128496440e-01 3.23697579756794572361511264091405e-03
            2.67714629312019669615807515583583e-01 1.46257414824437714795379861243418e-01 3.36140550758070926970488834228945e-03
            1.99727347669159671639249609143008e-01 1.59836334262202284239862137837918e-01 3.31664655306843515064629812627572e-03
            1.39516911332385140642742271666066e-01 1.71862005013749069570749838931079e-01 3.09013410543948529099478328419082e-03
            8.86426717314284795534717886766884e-02 1.82022981953933460363970198159222e-01 2.68387962170634321659723475761439e-03
            4.84220481925909940201790959690697e-02 1.90056140414945257122170119146176e-01 2.11482453904308486530849719997605e-03
            1.98959239325848247226247167418478e-02 1.95753587552677132954670469189296e-01 1.41380240215248842092221615018843e-03
            3.79657807820771298068507348943967e-03 1.98969067199380372779060621724057e-01 6.24067238582461905789522393916968e-04
            9.96203421921792342530466157768387e-01 5.29686847103782651458026009549940e-04 2.06086461459862931320857412642589e-06
            9.80104076067414897721619126969017e-01 2.77581785517834727511354842022229e-03 2.48687755885035024138937642668168e-05
            9.51577951807409005979820904030930e-01 6.75569460421819763251694013206361e-03 9.32495828035644834215836529445198e-05
            9.11357328268571409424225748807658e-01 1.23671517722194531513357773633288e-02 2.26199141214001111805514954156138e-04
            8.60483088667614692823804034560453e-01 1.94649685477286393808782349879039e-02 4.34145808734783986929800514431577e-04
            8.00272652330840550405355315888301e-01 2.78653426554105708434772026294013e-02 7.17253246402663083118589337772164e-04
            7.32285370687980496917646178189898e-01 3.73507182001073800847734673880041e-02 1.06484507069085619114512386573779e-03
            6.58282049981815053918410285405116e-01 4.76754329333715295979878590060252e-02 1.45602557470847319975870703956389e-03
            5.80179322820112863823283078090753e-01 5.85720841936082023404530616517150e-02 1.86144506304451013388445712593011e-03
            5.00000000000000111022302462515654e-01 6.97584556661925564435833280185761e-02 2.24603809001250797824922322831753e-03
            4.19820677179887136176716921909247e-01 8.09448271387769452411831139215792e-02 2.57246008795621603285064260546733e-03
            3.41717950018184946081589714594884e-01 9.18414783990136179836483165672689e-02 2.80487314199924031923871758920086e-03
            2.67714629312019669615807515583583e-01 1.02166193132277746680180996463605e-01 2.91269277782839211901610987354161e-03
            1.99727347669159671639249609143008e-01 1.11651568676974524696454693639680e-01 2.87390867895753613150966465639158e-03
            1.39516911332385140642742271666066e-01 1.20051942784656529017439652307075e-01 2.67763329093631454597135288508980e-03
            8.86426717314284795534717886766884e-02 1.27149759560165720451152537862072e-01 2.32560956215341861630374609148930e-03
            4.84220481925909940201790959690697e-02 1.32761216728166930867160999696353e-01 1.83251742384346952285945064886619e-03
            1.98959239325848247226247167418478e-02 1.36741093477206809847501745025511e-01 1.22507446267314419607874498296951e-03
            3.79657807820771298068507348943967e-03 1.38987224485281363195454673586937e-01 5.40760742671211468331204930137801e-04
            9.96203421921792342530466157768387e-01 3.36538824289303909690490579365019e-04 1.69000822087614555443309781457284e-06
            9.80104076067414897721619126969017e-01 1.76362785394960789112084320606755e-03 2.03935934898277234196897755857947e-05
            9.51577951807409005979820904030930e-01 4.29225972249925279144999734626253e-03 7.64691481502230083200366550499893e-05
            9.11357328268571409424225748807658e-01 7.85752325168580408043528251482712e-03 1.85494187972768884153929036529007e-04
            8.60483088667614692823804034560453e-01 1.23671517722194444777183974792933e-02 3.56020468604922792500017125760792e-04
            8.00272652330840550405355315888301e-01 1.77043657152261835197037953548715e-02 5.88182199977597009593610621891457e-04
            7.32285370687980496917646178189898e-01 2.37309400038064055582509581654449e-02 8.73224233498460177503586177039097e-04
            6.58282049981815053918410285405116e-01 3.02907920681986519984008054962032e-02 1.19401108332508059002574629658966e-03
            5.80179322820112863823283078090753e-01 3.72140264733227393856296316698717e-02 1.52647458594324953590204430042832e-03
            5.00000000000000111022302462515654e-01 4.43213358657142328378419904311158e-02 1.84185938738210767257053035450554e-03
            4.19820677179887136176716921909247e-01 5.14286452581057401678421570068167e-02 2.10954114390890555899571445763740e-03
            3.41717950018184946081589714594884e-01 5.83518796632298275550709831804852e-02 2.30013103962029355487040049865755e-03
            2.67714629312019669615807515583583e-01 6.49117317276220601174330226967868e-02 2.38854833284390813755160287712442e-03
            1.99727347669159671639249609143008e-01 7.09383060162022682781923776929034e-02 2.35674350419737190648339364429376e-03
            1.39516911332385140642742271666066e-01 7.62755199592090593618820548726944e-02 2.19578823476249823665185090248997e-03
            8.86426717314284795534717886766884e-02 8.07851484797426910855477899531252e-02 1.90711182614628459164318741159150e-03
            4.84220481925909940201790959690697e-02 8.43504120089292319661922192608472e-02 1.50275252884450182151720998291466e-03
            1.98959239325848247226247167418478e-02 8.68790438774788842390961463024723e-02 1.00462005045696099736485251696649e-03
            3.79657807820771298068507348943967e-03 8.83061329071391787337574896810111e-02 4.43449848266439894221846440558465e-04
            9.96203421921792342530466157768387e-01 1.83838086669908389340563625857783e-04 1.27539422074654254255572823073539e-06
            9.80104076067414897721619126969017e-01 9.63401387499757664613364571692955e-04 1.53903815116925176374740313889333e-05
            9.51577951807409005979820904030930e-01 2.34469475116560457580994203397040e-03 5.77087782245487039751476177062983e-05
            9.11357328268571409424225748807658e-01 4.29225972249926146506737723029801e-03 1.39986428704999896341018561507497e-04
            8.60483088667614692823804034560453e-01 6.75569460421820630613432001609908e-03 2.68677064713208468144101104257970e-04
            8.00272652330840550405355315888301e-01 9.67120725421441278235601402002430e-03 4.43881970117582756729657234018305e-04
            7.32285370687980496917646178189898e-01 1.29632906824082416025190056529937e-02 6.58993919119749030181842908149292e-04
            6.58282049981815053918410285405116e-01 1.65466830440539509894382064203455e-02 9.01081318048647071650147921673124e-04
            5.80179322820112863823283078090753e-01 2.03285770626506802349897640169729e-02 1.15198070694543094547035444463745e-03
            5.00000000000000111022302462515654e-01 2.42110240962954900711956440773065e-02 1.38999135570895201996499679353292e-03
            4.19820677179887136176716921909247e-01 2.80934711299403137851893319520968e-02 1.59200206847138472573965906065041e-03
            3.41717950018184946081589714594884e-01 3.18753651485370395612939375951100e-02 1.73583406202048890626388200075780e-03
            2.67714629312019669615807515583583e-01 3.54587575101827437440427104320406e-02 1.80255971660516845292421272972661e-03
            1.99727347669159671639249609143008e-01 3.87508409383765656253117981577816e-02 1.77855764717937741767328496678147e-03
            1.39516911332385140642742271666066e-01 4.16663535883727981223856318138132e-02 1.65708994193382720509777339401580e-03
            8.86426717314284795534717886766884e-02 4.41297884700917394940056226460001e-02 1.43923524828972544849226000707176e-03
            4.84220481925909940201790959690697e-02 4.60773534414253907454117609177047e-02 1.13407844223793892977913966291226e-03
            1.98959239325848247226247167418478e-02 4.74586468050912438365607215473574e-02 7.58154067282971436299299217154157e-04
            3.79657807820771298068507348943967e-03 4.82382101059210879889782574991841e-02 3.34657173073832912065656008948622e-04
            9.96203421921792342530466157768387e-01 7.55364286481397320671682393822266e-05 8.27810621346007799167614985696995e-07
            9.80104076067414897721619126969017e-01 3.95847789131206025028258954634452e-04 9.98932022327091967525865279231567e-06
            9.51577951807409005979820904030930e-01 9.63401387499746931011856965199058e-04 3.74566065786464646026991676119877e-05
            9.11357328268571409424225748807658e-01 1.76362785394959141124782142640015e-03 9.08599479606107709832307772757076e-05
            8.60483088667614692823804034560453e-01 2.77581785517831995321880178551055e-03 1.74388219943065453767030992260345e-04
            8.00272652330840550405355315888301e-01 3.97376011648251786789742823202687e-03 2.88107162091609568868588731049840e-04
            7.32285370687980496917646178189898e-01 5.32642990043208364170368440682068e-03 4.27728271600950871698942901133478e-04
            6.58282049981815053918410285405116e-01 6.79879433996063119172159616709905e-03 5.84858135346190259086507357011442e-04
            5.80179322820112863823283078090753e-01 8.35272025849728358415369200429268e-03 7.47707531744118262563569832224175e-04
            5.00000000000000111022302462515654e-01 9.94796196629241062658888239411681e-03 9.02191329643538047697304271821395e-04
            4.19820677179887136176716921909247e-01 1.15432036740875411384710247375551e-02 1.03330891739027760228841756884322e-03
            3.41717950018184946081589714594884e-01 1.30971295926241935309031205747488e-02 1.12666487746325384364243227253155e-03
            2.67714629312019669615807515583583e-01 1.45694940321527393461975563582200e-02 1.16997400077818508243310269278936e-03
            1.99727347669159671639249609143008e-01 1.59221638161022999158333846025926e-02 1.15439515646342588314765542634177e-03
            1.39516911332385140642742271666066e-01 1.71201060774065078051719979157497e-02 1.07555501831858228538318922318240e-03
            8.86426717314284795534717886766884e-02 1.81322960786352363471429782748601e-02 9.34153695986177502567915365006002e-04
            4.84220481925909940201790959690697e-02 1.89325225450850780084532942737496e-02 7.36087842216016832036507011594040e-04
            1.98959239325848247226247167418478e-02 1.95000761434536221128333011165523e-02 4.92089409928605299622550450777680e-04
            3.79657807820771298068507348943967e-03 1.98203875039366858579192864908691e-02 2.17213437126905753362521300786625e-04
            9.96203421921792342530466157768387e-01 1.44140051039273714401185363942481e-05 3.59499118241754465868904129186023e-07
            9.80104076067414897721619126969017e-01 7.55364286481405858763790717169684e-05 4.33813207936528044347552351966968e-06
            9.51577951807409005979820904030930e-01 1.83838086669908389340563625857783e-04 1.62665429629989633593303149927678e-05
            9.11357328268571409424225748807658e-01 3.36538824289304560211794070667679e-04 3.94583861731804972023188171625918e-05
            8.60483088667614692823804034560453e-01 5.29686847103783301979329500852600e-04 7.57327940529975890448025066525872e-05
            8.00272652330840550405355315888301e-01 7.58280469779300920173115141409426e-04 1.25118315784179774197495960130766e-04
            7.32285370687980496917646178189898e-01 1.01639949286151730183103136795353e-03 1.85752553207866688642774244399902e-04
            6.58282049981815053918410285405116e-01 1.29735887796911990646475665300841e-03 2.53990440001360432645954867680871e-04
            5.80179322820112863823283078090753e-01 1.59388197975947657446249561274954e-03 3.24712188311457035198859255586967e-04
            5.00000000000000111022302462515654e-01 1.89828903910385605666166775051806e-03 3.91800949551530052449022090144126e-04
            4.19820677179887136176716921909247e-01 2.20269609844823662306301237379103e-03 4.48742303002993702006062637366313e-04
            3.41717950018184946081589714594884e-01 2.49921920023859307422031683643127e-03 4.89284649843478819213060670989535e-04
            2.67714629312019669615807515583583e-01 2.78017858534619546201360762438526e-03 5.08092806252761521046412696023253e-04
            1.99727347669159671639249609143008e-01 3.03829760842841108473000311107626e-03 5.01327272385516573525998662574921e-04
            1.39516911332385140642742271666066e-01 3.26689123110393032922704747988973e-03 4.67088813232813065100257343686962e-04
            8.86426717314284795534717886766884e-02 3.46003925391840912520469153434988e-03 4.05681470314135802189559054653500e-04
            4.84220481925909940201790959690697e-02 3.61273999153780478107989004854517e-03 3.19666024331580558588000728903467e-04
            1.98959239325848247226247167418478e-02 3.72104164955957324861790525005745e-03 2.13703115668886697029252297319601e-04
            3.79657807820771298068507348943967e-03 3.78216407310378579728626924350010e-03 9.43308011564449364711945622552491e-05
            ];
        
    case  36
        
        % ALG. DEG.:   36
        % PTS CARD.:  361
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.8e-15
        
        xyw=[
            9.96203421921792342530466157768387e-01 3.78216407310378579728626924350010e-03 3.59499118241727625512388008544029e-07
            9.80104076067414897721619126969017e-01 1.98203875039369079025242115221772e-02 4.33813207936495602985672512263982e-06
            9.51577951807409005979820904030930e-01 4.82382101059210879889782574991841e-02 1.62665429629977470200180578174809e-05
            9.11357328268571409424225748807658e-01 8.83061329071393452672111834544921e-02 3.94583861731775495276623721974119e-05
            8.60483088667614692823804034560453e-01 1.38987224485281529728908367360418e-01 7.57327940529919240884512698919195e-05
            8.00272652330840550405355315888301e-01 1.98969067199380095223304465434921e-01 1.25118315784170422953758272655023e-04
            7.32285370687980496917646178189898e-01 2.66698229819158028064407517376821e-01 1.85752553207852810854966429943147e-04
            6.58282049981815053918410285405116e-01 3.40420591140215833547699730843306e-01 2.53990440001341459107936371353276e-04
            5.80179322820112863823283078090753e-01 4.18226795200127676732648751567467e-01 3.24712188311432803280304204562867e-04
            5.00000000000000111022302462515654e-01 4.98101710960896060242930616368540e-01 3.91800949551500778990364981524408e-04
            4.19820677179887136176716921909247e-01 5.77976626721664610286666174943093e-01 4.48742303002960200158932835279302e-04
            3.41717950018184946081589714594884e-01 6.55782830781576508982766426925082e-01 4.89284649843442281599847909490109e-04
            2.67714629312019669615807515583583e-01 7.29505192102634203443756177875912e-01 5.08092806252723573970375703368063e-04
            1.99727347669159671639249609143008e-01 7.97234354722411997506981151673244e-01 5.01327272385479168551047912671947e-04
            1.39516911332385140642742271666066e-01 8.57216197436511007090587099810364e-01 4.67088813232778153790303310444187e-04
            8.86426717314284795534717886766884e-02 9.07897289014653163796708668087376e-01 4.05681470314105498738838084804570e-04
            4.84220481925909940201790959690697e-02 9.47965211815871233724806188547518e-01 3.19666024331556651930097423530697e-04
            1.98959239325848247226247167418478e-02 9.76383034417855566466926120483549e-01 2.13703115668870732152262448266811e-04
            3.79657807820771298068507348943967e-03 9.92421257848688620484267630672548e-01 9.43308011564378891570734064764281e-05
            9.96203421921792342530466157768387e-01 3.72104164955957238125616726165390e-03 8.27810621346032892518677394344540e-07
            9.80104076067414897721619126969017e-01 1.95000761434538337490973702870178e-02 9.98932022327122291305376983183706e-06
            9.51577951807409005979820904030930e-01 4.74586468050912368976668176401290e-02 3.74566065786475962387166993572407e-05
            9.11357328268571409424225748807658e-01 8.68790438774790368947620322614966e-02 9.08599479606135356987706153120143e-05
            8.60483088667614692823804034560453e-01 1.36741093477206948625379823170078e-01 1.74388219943070766357676171232072e-04
            8.00272652330840550405355315888301e-01 1.95753587552676827643338697271247e-01 2.88107162091618350906185863635756e-04
            7.32285370687980496917646178189898e-01 2.62388199411587363929498906145454e-01 4.27728271600963827914904102911464e-04
            6.58282049981815053918410285405116e-01 3.34919155678224222949523891657009e-01 5.84858135346208040002136119284160e-04
            5.80179322820112863823283078090753e-01 4.11467956921389710345238199806772e-01 7.47707531744140922388974779266846e-04
            5.00000000000000111022302462515654e-01 4.90052038033707337838507100968855e-01 9.02191329643565369592050906533132e-04
            4.19820677179887136176716921909247e-01 5.68636119146025187376380927162245e-01 1.03330891739030904415141964847180e-03
            3.41717950018184946081589714594884e-01 6.45184920389190730283246466569835e-01 1.12666487746328788759064831737078e-03
            2.67714629312019669615807515583583e-01 7.17715876655827478280968989565736e-01 1.16997400077822064426436021733480e-03
            1.99727347669159671639249609143008e-01 7.84350488514737764766948657779722e-01 1.15439515646346101129804395668543e-03
            1.39516911332385140642742271666066e-01 8.43362982590208143385268613201333e-01 1.07555501831861481144836378831542e-03
            8.86426717314284795534717886766884e-02 8.93225032189935985726947365037631e-01 9.34153695986205800244617236671729e-04
            4.84220481925909940201790959690697e-02 9.32645429262323633068376693699975e-01 7.36087842216039166601260212985380e-04
            1.98959239325848247226247167418478e-02 9.60603999923961171525377267244039e-01 4.92089409928620261612530750738870e-04
            3.79657807820771298068507348943967e-03 9.76383034417855344422321195452241e-01 2.17213437126912366995773462363672e-04
            9.96203421921792342530466157768387e-01 3.61273999153780478107989004854517e-03 1.27539422074654381310514911218590e-06
            9.80104076067414897721619126969017e-01 1.89325225450852896447173634442152e-02 1.53903815116925311900011874577388e-05
            9.51577951807409005979820904030930e-01 4.60773534414253907454117609177047e-02 5.77087782245487581852562419815200e-05
            9.11357328268571409424225748807658e-01 8.43504120089293846218581052198715e-02 1.39986428705000031866290122195551e-04
            8.60483088667614692823804034560453e-01 1.32761216728167097400614693469834e-01 2.68677064713208739194644225634079e-04
            8.00272652330840550405355315888301e-01 1.90056140414944979566413962857041e-01 4.43881970117583190410526228220078e-04
            7.32285370687980496917646178189898e-01 2.54751338629611245867323532365845e-01 6.58993919119749680703146399451953e-04
            6.58282049981815053918410285405116e-01 3.25171266974131012439386267942609e-01 9.01081318048648047432103158627115e-04
            5.80179322820112863823283078090753e-01 3.99492100117236459411174109845888e-01 1.15198070694543202967252693014188e-03
            5.00000000000000111022302462515654e-01 4.75788975903704391967607989499811e-01 1.38999135570895332100760377613824e-03
            4.19820677179887136176716921909247e-01 5.52085851690172546568646794185042e-01 1.59200206847138646046313503745750e-03
            3.41717950018184946081589714594884e-01 6.26406684833277993540434636088321e-01 1.73583406202049064098735797756490e-03
            2.67714629312019669615807515583583e-01 6.96826613177797593579043677891605e-01 1.80255971660517018764768870653370e-03
            1.99727347669159671639249609143008e-01 7.61521811392463776613226400513668e-01 1.77855764717937936923719544068945e-03
            1.39516911332385140642742271666066e-01 8.18816735079242130623811135592405e-01 1.65708994193382893982124937082290e-03
            8.86426717314284795534717886766884e-02 8.67227539798479774013628684770083e-01 1.43923524828972696637530148677797e-03
            4.84220481925909940201790959690697e-02 9.05500598365983622173303047020454e-01 1.13407844223794001398131214841669e-03
            1.98959239325848247226247167418478e-02 9.32645429262323855112981618731283e-01 7.58154067282972195240819957007261e-04
            3.79657807820771298068507348943967e-03 9.47965211815871233724806188547518e-01 3.34657173073833237326307754599952e-04
            9.96203421921792342530466157768387e-01 3.46003925391840869152382254014810e-03 1.69000822087614873080665001819911e-06
            9.80104076067414897721619126969017e-01 1.81322960786354375750661915844830e-02 2.03935934898277640772712437922110e-05
            9.51577951807409005979820904030930e-01 4.41297884700917325551117187387717e-02 7.64691481502231573978353718068490e-05
            9.11357328268571409424225748807658e-01 8.07851484797428298634258680976927e-02 1.85494187972769263624689406455559e-04
            8.60483088667614692823804034560453e-01 1.27149759560165859229030616006639e-01 3.56020468604923497231429241338674e-04
            8.00272652330840550405355315888301e-01 1.82022981953933182808214041870087e-01 5.88182199977598202216000355946335e-04
            7.32285370687980496917646178189898e-01 2.43983689308213069768527248015744e-01 8.73224233498461803806844905295748e-04
            6.58282049981815053918410285405116e-01 3.11427157949986266327613293469767e-01 1.19401108332508297527052576469941e-03
            5.80179322820112863823283078090753e-01 3.82606650706564355157723866796005e-01 1.52647458594325257166812725984073e-03
            5.00000000000000111022302462515654e-01 4.55678664134285593689810411888175e-01 1.84185938738211114201748230811972e-03
            4.19820677179887136176716921909247e-01 5.28750677562007109777653113269480e-01 2.10954114390890946212353540545337e-03
            3.41717950018184946081589714594884e-01 5.99930170318585198607763686595717e-01 2.30013103962029789167909044067528e-03
            2.67714629312019669615807515583583e-01 6.67373638960358284144547269534087e-01 2.38854833284391290804116181334393e-03
            1.99727347669159671639249609143008e-01 7.29334346314638004571406781906262e-01 2.35674350419737624329208358631149e-03
            1.39516911332385140642742271666066e-01 7.84207568708405799995375673461240e-01 2.19578823476250257346054084450770e-03
            8.86426717314284795534717886766884e-02 8.30572179788828690583102343225619e-01 1.90711182614628849477100835940746e-03
            4.84220481925909940201790959690697e-02 8.67227539798479662991326222254429e-01 1.50275252884450485728329294232708e-03
            1.98959239325848247226247167418478e-02 8.93225032189936096749249827553285e-01 1.00462005045696294892876299087447e-03
            3.79657807820771298068507348943967e-03 9.07897289014653052774406205571722e-01 4.43449848266440761583584428962013e-04
            9.96203421921792342530466157768387e-01 3.26689123110392946186530949148619e-03 2.06086461459863778353804666942928e-06
            9.80104076067414897721619126969017e-01 1.71201060774066951553074034109159e-02 2.48687755885036074459792238000588e-05
            9.51577951807409005979820904030930e-01 4.16663535883727911834917279065849e-02 9.32495828035648764448711789398772e-05
            9.11357328268571409424225748807658e-01 7.62755199592091842619723252028052e-02 2.26199141214002060482415878972517e-04
            8.60483088667614692823804034560453e-01 1.20051942784656640039742114822729e-01 4.34145808734785775863385115513893e-04
            8.00272652330840550405355315888301e-01 1.71862005013748792014993682641943e-01 7.17253246402666118884672297184579e-04
            7.32285370687980496917646178189898e-01 2.30363911111912067486429123164271e-01 1.06484507069086074479424830485641e-03
            6.58282049981815053918410285405116e-01 2.94042517084813381789132336052717e-01 1.45602557470847927129087295838872e-03
            5.80179322820112863823283078090753e-01 3.61248592986278871386218725092476e-01 1.86144506304451794014009902156204e-03
            5.00000000000000111022302462515654e-01 4.30241544333807235389599554764573e-01 2.24603809001251751922834110075655e-03
            4.19820677179887136176716921909247e-01 4.99234495681335821437585309467977e-01 2.57246008795622730855323645471344e-03
            3.41717950018184946081589714594884e-01 5.66440571582801366545822929765563e-01 2.80487314199925202862218043264875e-03
            2.67714629312019669615807515583583e-01 6.30119177555702569826223680138355e-01 2.91269277782840426208044171119127e-03
            1.99727347669159671639249609143008e-01 6.88621083653865651008629811258288e-01 2.87390867895754827457399649404124e-03
            1.39516911332385140642742271666066e-01 7.40431145882958219317515613511205e-01 2.67763329093632625535481572853769e-03
            8.86426717314284795534717886766884e-02 7.84207568708405688973073210945586e-01 2.32560956215342859096373295813009e-03
            4.84220481925909940201790959690697e-02 8.18816735079241908579206210561097e-01 1.83251742384347732911509254449811e-03
            1.98959239325848247226247167418478e-02 8.43362982590208143385268613201333e-01 1.22507446267314918340873841628991e-03
            3.79657807820771298068507348943967e-03 8.57216197436510785045982174779056e-01 5.40760742671213745155767149697112e-04
            9.96203421921792342530466157768387e-01 3.03829760842841195209174109947980e-03 2.37834958723482436934791410532153e-06
            9.80104076067414897721619126969017e-01 1.59221638161024837965218381441446e-02 2.86999164025493689049624290721141e-05
            9.51577951807409005979820904030930e-01 3.87508409383765795030996059722384e-02 1.07615078253876813424959324461128e-04
            9.11357328268571409424225748807658e-01 7.09383060162024209338582636519277e-02 2.61046082468628138934657600955802e-04
            8.60483088667614692823804034560453e-01 1.11651568676974691229908387413161e-01 5.01027819920718260442160651990662e-04
            8.00272652330840550405355315888301e-01 1.59836334262202117706408444064436e-01 8.27749164326762828548800232653093e-04
            7.32285370687980496917646178189898e-01 2.14244696467297629505210920797253e-01 1.22888898979900992695923900299704e-03
            6.58282049981815053918410285405116e-01 2.73467530210110443267268465206143e-01 1.68033251678964914467562241640053e-03
            5.80179322820112863823283078090753e-01 3.35971006830077856264438196376432e-01 2.14820860428740004199887891900289e-03
            5.00000000000000111022302462515654e-01 4.00136326165420164180375195428496e-01 2.59204982532795563318628317972525e-03
            4.19820677179887136176716921909247e-01 4.64301645500762694140917119511869e-01 2.96875852252929256502866905975679e-03
            3.41717950018184946081589714594884e-01 5.26805122120730051626935619424330e-01 3.23697579756793748367860175108035e-03
            2.67714629312019669615807515583583e-01 5.86027955863542837633417548204307e-01 3.36140550758070059608750845825398e-03
            1.99727347669159671639249609143008e-01 6.40436318068638210654341946792556e-01 3.31664655306842691070978723644203e-03
            1.39516911332385140642742271666066e-01 6.88621083653865984075537198805250e-01 3.09013410543947748473914138855889e-03
            8.86426717314284795534717886766884e-02 7.29334346314638226616011706937570e-01 2.68387962170633627770333085038601e-03
            4.84220481925909940201790959690697e-02 7.61521811392463998657831325544976e-01 2.11482453904307966113806926955476e-03
            1.98959239325848247226247167418478e-02 7.84350488514738208856158507842338e-01 1.41380240215248473463482969947336e-03
            3.79657807820771298068507348943967e-03 7.97234354722412219551586076704552e-01 6.24067238582460279486263665660317e-04
            9.96203421921792342530466157768387e-01 2.78017858534619589569447661858703e-03 2.63423808139417025888924343834940e-06
            9.80104076067414897721619126969017e-01 1.45694940321529041449277741548940e-02 3.17877628781744695093090802284763e-05
            9.51577951807409005979820904030930e-01 3.54587575101827506829366143392690e-02 1.19193468777719437145037884029364e-04
            9.11357328268571409424225748807658e-01 6.49117317276221988953111008413543e-02 2.89132234860866156687247974232946e-04
            8.60483088667614692823804034560453e-01 1.02166193132277885458059074608173e-01 5.54933795333068034631718301596948e-04
            8.00272652330840550405355315888301e-01 1.46257414824437548261926167469937e-01 9.16807344981985204712682246963595e-04
            7.32285370687980496917646178189898e-01 1.96043506564347491671895795661840e-01 1.36110611833908804260695468713038e-03
            6.58282049981815053918410285405116e-01 2.50235055699803365936162435900769e-01 1.86112080784495972571312982779546e-03
            5.80179322820112863823283078090753e-01 3.07428540211152623307100384408841e-01 2.37933604990836185602565144847631e-03
            5.00000000000000111022302462515654e-01 3.66142685343990192947671857837122e-01 2.87093049541494674953279186979671e-03
            4.19820677179887136176716921909247e-01 4.24856830476827873610545793781057e-01 3.28816957628273731639723109765328e-03
            3.41717950018184946081589714594884e-01 4.82050314988177130981483742289129e-01 3.58524455793673242284813618141470e-03
            2.67714629312019669615807515583583e-01 5.36241864123632949734599151270231e-01 3.72306175786886808798392678454547e-03
            1.99727347669159671639249609143008e-01 5.86027955863542726611115085688652e-01 3.67348715239772553517449971138831e-03
            1.39516911332385140642742271666066e-01 6.30119177555702791870828605169663e-01 3.42260405318617648082191173841693e-03
            8.86426717314284795534717886766884e-02 6.67373638960358395166849732049741e-01 2.97264033148149755050870801653673e-03
            4.84220481925909940201790959690697e-02 6.96826613177797704601346140407259e-01 2.34236016694719384467759937251685e-03
            1.98959239325848247226247167418478e-02 7.17715876655827700325573914597044e-01 1.56591450950095993285837714381614e-03
            3.79657807820771298068507348943967e-03 7.29505192102634314466058640391566e-01 6.91211121379232788786151076010356e-04
            9.96203421921792342530466157768387e-01 2.49921920023859307422031683643127e-03 2.82190191218257094781548914574909e-06
            9.80104076067414897721619126969017e-01 1.30971295926243392476751026265447e-02 3.40523316717264792636427062078042e-05
            9.51577951807409005979820904030930e-01 3.18753651485370395612939375951100e-02 1.27684843613492084515859126270243e-04
            9.11357328268571409424225748807658e-01 5.83518796632299385773734456961392e-02 3.09730093187204240986337699936826e-04
            8.60483088667614692823804034560453e-01 9.18414783990137290059507790829230e-02 5.94467428455186259333986598107913e-04
            8.00272652330840550405355315888301e-01 1.31476927861084946780323434722959e-01 9.82120947298144829351596030164728e-04
            7.32285370687980496917646178189898e-01 1.76231734993637922936926543115987e-01 1.45807168499806566810961250268974e-03
            6.58282049981815053918410285405116e-01 2.24946792653554189511666550060909e-01 1.99370755572742149042131387659538e-03
            5.80179322820112863823283078090753e-01 2.76360415998729902486985565701616e-01 2.54884059128315885317528355358263e-03
            5.00000000000000111022302462515654e-01 3.29141024990907471448053911444731e-01 3.07545635756164593976880183845424e-03
            4.19820677179887136176716921909247e-01 3.81921633983085151431424719703500e-01 3.52241966298731754708706276346675e-03
            3.41717950018184946081589714594884e-01 4.33335257328260836651168119715294e-01 3.84065834638959088809251340990158e-03
            2.67714629312019669615807515583583e-01 4.82050314988177075470332511031302e-01 3.98829368078363038357947445433638e-03
            1.99727347669159671639249609143008e-01 5.26805122120729940604633156908676e-01 3.93518736705944505160914559382945e-03
            1.39516911332385140642742271666066e-01 5.66440571582801477568125392281218e-01 3.66643128825252595862083104805151e-03
            8.86426717314284795534717886766884e-02 5.99930170318585198607763686595717e-01 3.18441202975816746445492100292540e-03
            4.84220481925909940201790959690697e-02 6.26406684833277993540434636088321e-01 2.50923053645564107447252233384916e-03
            1.98959239325848247226247167418478e-02 6.45184920389190841305548929085489e-01 1.67747068113771710846926499272058e-03
            3.79657807820771298068507348943967e-03 6.55782830781576508982766426925082e-01 7.40453187932655731810782651081126e-04
            9.96203421921792342530466157768387e-01 2.20269609844823662306301237379103e-03 2.93648036621551472048763375966018e-06
            9.80104076067414897721619126969017e-01 1.15432036740876695080082470212801e-02 3.54349677946616507134315610105801e-05
            9.51577951807409005979820904030930e-01 2.80934711299403137851893319520968e-02 1.32869266190872436107089926693448e-04
            9.11357328268571409424225748807658e-01 5.14286452581058303734629078007856e-02 3.22306148751595310936179927807643e-04
            8.60483088667614692823804034560453e-01 8.09448271387770423856977686227765e-02 6.18604751808375890979874167641128e-04
            8.00272652330840550405355315888301e-01 1.15877677319350141926790342949971e-01 1.02199827234937469684628030819340e-03
            7.32285370687980496917646178189898e-01 1.55322492343285017968312899938610e-01 1.51727416783953121537376063088232e-03
            6.58282049981815053918410285405116e-01 1.98257688837027712391858358387253e-01 2.07465860811618637085662619767845e-03
            5.80179322820112863823283078090753e-01 2.43571276192108138181069421079883e-01 2.65233186192755966692491398362108e-03
            5.00000000000000111022302462515654e-01 2.90089661410056376400490307787550e-01 3.20032995198883717449800379029057e-03
            4.19820677179887136176716921909247e-01 3.36608046628004753397789272639784e-01 3.66544143057532514559659908570666e-03
            3.41717950018184946081589714594884e-01 3.81921633983085151431424719703500e-01 3.99660164615428178946920922953723e-03
            2.67714629312019669615807515583583e-01 4.24856830476827762588243331265403e-01 4.15023145835156318655956297902776e-03
            1.99727347669159671639249609143008e-01 4.64301645500762583118614656996215e-01 4.09496885446731465757475021405298e-03
            1.39516911332385140642742271666066e-01 4.99234495681335932459887771983631e-01 3.81530039919234257791980091667483e-03
            8.86426717314284795534717886766884e-02 5.28750677562007109777653113269480e-01 3.31370958110055197770593693462615e-03
            4.84220481925909940201790959690697e-02 5.52085851690172546568646794185042e-01 2.61111350922593613357869379854037e-03
            1.98959239325848247226247167418478e-02 5.68636119146025298398683389677899e-01 1.74558148133973104962346489799074e-03
            3.79657807820771298068507348943967e-03 5.77976626721664610286666174943093e-01 7.70518010948233081185143600322363e-04
            9.96203421921792342530466157768387e-01 1.89828903910385692402340573892161e-03 2.97500579217638840850046651198557e-06
            9.80104076067414897721619126969017e-01 9.94796196629252511833829686338504e-03 3.58998601344522795682755178514611e-05
            9.51577951807409005979820904030930e-01 2.42110240962955039489834518917633e-02 1.34612456826847690051149331047498e-04
            9.11357328268571409424225748807658e-01 4.43213358657143299823566451323131e-02 3.26534674102325569479410294349009e-04
            8.60483088667614692823804034560453e-01 6.97584556661926674658857905342302e-02 6.26720594106873062252571759955799e-04
            8.00272652330840550405355315888301e-01 9.98636738345797247973223420558497e-02 1.03540647327812112069023164906412e-03
            7.32285370687980496917646178189898e-01 1.33857314656009779296752526533965e-01 1.53718018672117017961453733221333e-03
            6.58282049981815053918410285405116e-01 1.70858975009092500796370472926355e-01 2.10187728375271949687874695200662e-03
            5.80179322820112863823283078090753e-01 2.09910338589943623599509692212450e-01 2.68712937528605907977485856008570e-03
            5.00000000000000111022302462515654e-01 2.50000000000000000000000000000000e-01 3.24231697701179381182479133371999e-03
            4.19820677179887136176716921909247e-01 2.90089661410056487422792770303204e-01 3.71353052869163261645590878856638e-03
            3.41717950018184946081589714594884e-01 3.29141024990907582470356373960385e-01 4.04903543137058018891449862053378e-03
            2.67714629312019669615807515583583e-01 3.66142685343990248458823089094949e-01 4.20468080410873416147277126242443e-03
            1.99727347669159671639249609143008e-01 4.00136326165420275202677657944150e-01 4.14869317737780012034232157702718e-03
            1.39516911332385140642742271666066e-01 4.30241544333807568456506942311535e-01 3.86535558591814733947478188724745e-03
            8.86426717314284795534717886766884e-02 4.55678664134285871245566568177310e-01 3.35718410066189161880534364001960e-03
            4.84220481925909940201790959690697e-02 4.75788975903704614012212914531119e-01 2.64537025459105675645465893808250e-03
            1.98959239325848247226247167418478e-02 4.90052038033707670905414488515817e-01 1.76848279915262540824827386387597e-03
            3.79657807820771298068507348943967e-03 4.98101710960896282287535541399848e-01 7.80626893310883683031742297231403e-04
            9.96203421921792342530466157768387e-01 1.59388197975947657446249561274954e-03 2.93648036621552742598184257416527e-06
            9.80104076067414897721619126969017e-01 8.35272025849737725922139475187578e-03 3.54349677946618065674938558018425e-05
            9.51577951807409005979820904030930e-01 2.03285770626506802349897640169729e-02 1.32869266190873005313230481583275e-04
            9.11357328268571409424225748807658e-01 3.72140264733228157134625746493839e-02 3.22306148751596720399004158963407e-04
            8.60483088667614692823804034560453e-01 5.85720841936082717293921007239987e-02 6.18604751808378601485305381402213e-04
            8.00272652330840550405355315888301e-01 8.38496703498092521567031099039013e-02 1.02199827234937925049540474731202e-03
            7.32285370687980496917646178189898e-01 1.12392136968734485114040921871492e-01 1.51727416783953793742723004100981e-03
            6.58282049981815053918410285405116e-01 1.43460261181157233689731356207631e-01 2.07465860811619504447400608171392e-03
            5.80179322820112863823283078090753e-01 1.76249400987778997995647500829364e-01 2.65233186192757137630837682706897e-03
            5.00000000000000111022302462515654e-01 2.09910338589943512577207229696796e-01 3.20032995198885105228581160474732e-03
            4.19820677179887136176716921909247e-01 2.43571276192108138181069421079883e-01 3.66544143057534075810788287697051e-03
            3.41717950018184946081589714594884e-01 2.76360415998729902486985565701616e-01 3.99660164615429913670396899760817e-03
            2.67714629312019669615807515583583e-01 3.07428540211152623307100384408841e-01 4.15023145835158140115606073550225e-03
            1.99727347669159671639249609143008e-01 3.35971006830077745242135733860778e-01 4.09496885446733200480950998212393e-03
            1.39516911332385140642742271666066e-01 3.61248592986278982408521187608130e-01 3.81530039919235949147369169054400e-03
            8.86426717314284795534717886766884e-02 3.82606650706564410668875098053832e-01 3.31370958110056628917461374328468e-03
            4.84220481925909940201790959690697e-02 3.99492100117236459411174109845888e-01 2.61111350922594784296215664198826e-03
            1.98959239325848247226247167418478e-02 4.11467956921389821367540662322426e-01 1.74558148133973863903867229652178e-03
            3.79657807820771298068507348943967e-03 4.18226795200127676732648751567467e-01 7.70518010948236442211878305386108e-04
            9.96203421921792342530466157768387e-01 1.29735887796911990646475665300841e-03 2.82190191218257560649669904440096e-06
            9.80104076067414897721619126969017e-01 6.79879433996070665219280115820766e-03 3.40523316717265334737513304830259e-05
            9.51577951807409005979820904030930e-01 1.65466830440539509894382064203455e-02 1.27684843613492301356293623371130e-04
            9.11357328268571409424225748807658e-01 3.02907920681987109789989887076445e-02 3.09730093187204728877315318413821e-04
            8.60483088667614692823804034560453e-01 4.76754329333715851091390902638523e-02 5.94467428455187235115941835061903e-04
            8.00272652330840550405355315888301e-01 6.82504198080744611809578259453701e-02 9.82120947298146347234637509870936e-04
            7.32285370687980496917646178189898e-01 9.14828943183815940232150865085714e-02 1.45807168499806805335439197079950e-03
            6.58282049981815053918410285405116e-01 1.16771157364630742692135356719518e-01 1.99370755572742452618739683600779e-03
            5.80179322820112863823283078090753e-01 1.43460261181157233689731356207631e-01 2.54884059128316318998397349560037e-03
            5.00000000000000111022302462515654e-01 1.70858975009092445285219241668528e-01 3.07545635756165114393922976887552e-03
            4.19820677179887136176716921909247e-01 1.98257688837027712391858358387253e-01 3.52241966298732318493835968808980e-03
            3.41717950018184946081589714594884e-01 2.24946792653554189511666550060909e-01 3.84065834638959695962467932872642e-03
            2.67714629312019669615807515583583e-01 2.50235055699803310425011204642942e-01 3.98829368078363645511164037316121e-03
            1.99727347669159671639249609143008e-01 2.73467530210110387756117233948316e-01 3.93518736705945199050304950105783e-03
            1.39516911332385140642742271666066e-01 2.94042517084813437300283567310544e-01 3.66643128825253203015299696687634e-03
            8.86426717314284795534717886766884e-02 3.11427157949986266327613293469767e-01 3.18441202975817266862534893334669e-03
            4.84220481925909940201790959690697e-02 3.25171266974131012439386267942609e-01 2.50923053645564497760034328166512e-03
            1.98959239325848247226247167418478e-02 3.34919155678224278460675122914836e-01 1.67747068113771971055447895793122e-03
            3.79657807820771298068507348943967e-03 3.40420591140215833547699730843306e-01 7.40453187932656924433172385136004e-04
            9.96203421921792342530466157768387e-01 1.01639949286151795235233485925619e-03 2.63423808139417703515282147275212e-06
            9.80104076067414897721619126969017e-01 5.32642990043214609174881957187608e-03 3.17877628781745508244720166413089e-05
            9.51577951807409005979820904030930e-01 1.29632906824082502761363855370291e-02 1.19193468777719748853162473611889e-04
            9.11357328268571409424225748807658e-01 2.37309400038064645388491413768861e-02 2.89132234860866915628768714086050e-04
            8.60483088667614692823804034560453e-01 3.73507182001074494737125064602878e-02 5.54933795333069444094542532752712e-04
            8.00272652330840550405355315888301e-01 5.34699328447218805160368049200770e-02 9.16807344981987481537244466522907e-04
            7.32285370687980496917646178189898e-01 7.16711227476720530438214495916327e-02 1.36110611833909151205390664074457e-03
            6.58282049981815053918410285405116e-01 9.14828943183816495343663177663984e-02 1.86112080784496449620268876401497e-03
            5.80179322820112863823283078090753e-01 1.12392136968734554502979960943776e-01 2.37933604990836792755781736730114e-03
            5.00000000000000111022302462515654e-01 1.33857314656009807052328142162878e-01 2.87093049541495412210756477122686e-03
            4.19820677179887136176716921909247e-01 1.55322492343285101235039746825350e-01 3.28816957628274599001461098168875e-03
            3.41717950018184946081589714594884e-01 1.76231734993638033959229005631641e-01 3.58524455793674153014638505965195e-03
            2.67714629312019669615807515583583e-01 1.96043506564347574938622642548580e-01 3.72306175786887762896304465698449e-03
            1.99727347669159671639249609143008e-01 2.14244696467297712771937767683994e-01 3.67348715239773464247274858962555e-03
            1.39516911332385140642742271666066e-01 2.30363911111912289531034048195579e-01 3.42260405318618515443929162245240e-03
            8.86426717314284795534717886766884e-02 2.43983689308213236301980941789225e-01 2.97264033148150535676434991216865e-03
            4.84220481925909940201790959690697e-02 2.54751338629611412400777226139326e-01 2.34236016694719948252889629713991e-03
            1.98959239325848247226247167418478e-02 2.62388199411587585974103831176762e-01 1.56591450950096383598619809163210e-03
            3.79657807820771298068507348943967e-03 2.66698229819158139086709979892476e-01 6.91211121379234523509627052817450e-04
            9.96203421921792342530466157768387e-01 7.58280469779302004375287626913860e-04 2.37834958723483029857854488542390e-06
            9.80104076067414897721619126969017e-01 3.97376011648256817487823155943261e-03 2.86999164025494434438617874505439e-05
            9.51577951807409005979820904030930e-01 9.67120725421442666014382183448106e-03 1.07615078253877084475502445837236e-04
            9.11357328268571409424225748807658e-01 1.77043657152262425003019785663128e-02 2.61046082468628789455961092258462e-04
            8.60483088667614692823804034560453e-01 2.78653426554106437018631936552993e-02 5.01027819920719561484767634595983e-04
            8.00272652330840550405355315888301e-01 3.98910134069573249493423361400346e-02 8.27749164326764996953145203661961e-04
            7.32285370687980496917646178189898e-01 5.34699328447219221494002283634472e-02 1.22888898979901317956575645951034e-03
            6.58282049981815053918410285405116e-01 6.82504198080745583254724806465674e-02 1.68033251678965348148431235841826e-03
            5.80179322820112863823283078090753e-01 8.38496703498093631790055724195554e-02 2.14820860428740567985017584362595e-03
            5.00000000000000111022302462515654e-01 9.98636738345798080640491889425903e-02 2.59204982532796213839931809275186e-03
            4.19820677179887136176716921909247e-01 1.15877677319350308460244036723452e-01 2.96875852252930037128431095538872e-03
            3.41717950018184946081589714594884e-01 1.31476927861085113313777128496440e-01 3.23697579756794572361511264091405e-03
            2.67714629312019669615807515583583e-01 1.46257414824437714795379861243418e-01 3.36140550758070926970488834228945e-03
            1.99727347669159671639249609143008e-01 1.59836334262202284239862137837918e-01 3.31664655306843515064629812627572e-03
            1.39516911332385140642742271666066e-01 1.71862005013749069570749838931079e-01 3.09013410543948529099478328419082e-03
            8.86426717314284795534717886766884e-02 1.82022981953933460363970198159222e-01 2.68387962170634321659723475761439e-03
            4.84220481925909940201790959690697e-02 1.90056140414945257122170119146176e-01 2.11482453904308486530849719997605e-03
            1.98959239325848247226247167418478e-02 1.95753587552677132954670469189296e-01 1.41380240215248842092221615018843e-03
            3.79657807820771298068507348943967e-03 1.98969067199380372779060621724057e-01 6.24067238582461905789522393916968e-04
            9.96203421921792342530466157768387e-01 5.29686847103782651458026009549940e-04 2.06086461459862931320857412642589e-06
            9.80104076067414897721619126969017e-01 2.77581785517834727511354842022229e-03 2.48687755885035024138937642668168e-05
            9.51577951807409005979820904030930e-01 6.75569460421819763251694013206361e-03 9.32495828035644834215836529445198e-05
            9.11357328268571409424225748807658e-01 1.23671517722194531513357773633288e-02 2.26199141214001111805514954156138e-04
            8.60483088667614692823804034560453e-01 1.94649685477286393808782349879039e-02 4.34145808734783986929800514431577e-04
            8.00272652330840550405355315888301e-01 2.78653426554105708434772026294013e-02 7.17253246402663083118589337772164e-04
            7.32285370687980496917646178189898e-01 3.73507182001073800847734673880041e-02 1.06484507069085619114512386573779e-03
            6.58282049981815053918410285405116e-01 4.76754329333715295979878590060252e-02 1.45602557470847319975870703956389e-03
            5.80179322820112863823283078090753e-01 5.85720841936082023404530616517150e-02 1.86144506304451013388445712593011e-03
            5.00000000000000111022302462515654e-01 6.97584556661925564435833280185761e-02 2.24603809001250797824922322831753e-03
            4.19820677179887136176716921909247e-01 8.09448271387769452411831139215792e-02 2.57246008795621603285064260546733e-03
            3.41717950018184946081589714594884e-01 9.18414783990136179836483165672689e-02 2.80487314199924031923871758920086e-03
            2.67714629312019669615807515583583e-01 1.02166193132277746680180996463605e-01 2.91269277782839211901610987354161e-03
            1.99727347669159671639249609143008e-01 1.11651568676974524696454693639680e-01 2.87390867895753613150966465639158e-03
            1.39516911332385140642742271666066e-01 1.20051942784656529017439652307075e-01 2.67763329093631454597135288508980e-03
            8.86426717314284795534717886766884e-02 1.27149759560165720451152537862072e-01 2.32560956215341861630374609148930e-03
            4.84220481925909940201790959690697e-02 1.32761216728166930867160999696353e-01 1.83251742384346952285945064886619e-03
            1.98959239325848247226247167418478e-02 1.36741093477206809847501745025511e-01 1.22507446267314419607874498296951e-03
            3.79657807820771298068507348943967e-03 1.38987224485281363195454673586937e-01 5.40760742671211468331204930137801e-04
            9.96203421921792342530466157768387e-01 3.36538824289303909690490579365019e-04 1.69000822087614555443309781457284e-06
            9.80104076067414897721619126969017e-01 1.76362785394960789112084320606755e-03 2.03935934898277234196897755857947e-05
            9.51577951807409005979820904030930e-01 4.29225972249925279144999734626253e-03 7.64691481502230083200366550499893e-05
            9.11357328268571409424225748807658e-01 7.85752325168580408043528251482712e-03 1.85494187972768884153929036529007e-04
            8.60483088667614692823804034560453e-01 1.23671517722194444777183974792933e-02 3.56020468604922792500017125760792e-04
            8.00272652330840550405355315888301e-01 1.77043657152261835197037953548715e-02 5.88182199977597009593610621891457e-04
            7.32285370687980496917646178189898e-01 2.37309400038064055582509581654449e-02 8.73224233498460177503586177039097e-04
            6.58282049981815053918410285405116e-01 3.02907920681986519984008054962032e-02 1.19401108332508059002574629658966e-03
            5.80179322820112863823283078090753e-01 3.72140264733227393856296316698717e-02 1.52647458594324953590204430042832e-03
            5.00000000000000111022302462515654e-01 4.43213358657142328378419904311158e-02 1.84185938738210767257053035450554e-03
            4.19820677179887136176716921909247e-01 5.14286452581057401678421570068167e-02 2.10954114390890555899571445763740e-03
            3.41717950018184946081589714594884e-01 5.83518796632298275550709831804852e-02 2.30013103962029355487040049865755e-03
            2.67714629312019669615807515583583e-01 6.49117317276220601174330226967868e-02 2.38854833284390813755160287712442e-03
            1.99727347669159671639249609143008e-01 7.09383060162022682781923776929034e-02 2.35674350419737190648339364429376e-03
            1.39516911332385140642742271666066e-01 7.62755199592090593618820548726944e-02 2.19578823476249823665185090248997e-03
            8.86426717314284795534717886766884e-02 8.07851484797426910855477899531252e-02 1.90711182614628459164318741159150e-03
            4.84220481925909940201790959690697e-02 8.43504120089292319661922192608472e-02 1.50275252884450182151720998291466e-03
            1.98959239325848247226247167418478e-02 8.68790438774788842390961463024723e-02 1.00462005045696099736485251696649e-03
            3.79657807820771298068507348943967e-03 8.83061329071391787337574896810111e-02 4.43449848266439894221846440558465e-04
            9.96203421921792342530466157768387e-01 1.83838086669908389340563625857783e-04 1.27539422074654254255572823073539e-06
            9.80104076067414897721619126969017e-01 9.63401387499757664613364571692955e-04 1.53903815116925176374740313889333e-05
            9.51577951807409005979820904030930e-01 2.34469475116560457580994203397040e-03 5.77087782245487039751476177062983e-05
            9.11357328268571409424225748807658e-01 4.29225972249926146506737723029801e-03 1.39986428704999896341018561507497e-04
            8.60483088667614692823804034560453e-01 6.75569460421820630613432001609908e-03 2.68677064713208468144101104257970e-04
            8.00272652330840550405355315888301e-01 9.67120725421441278235601402002430e-03 4.43881970117582756729657234018305e-04
            7.32285370687980496917646178189898e-01 1.29632906824082416025190056529937e-02 6.58993919119749030181842908149292e-04
            6.58282049981815053918410285405116e-01 1.65466830440539509894382064203455e-02 9.01081318048647071650147921673124e-04
            5.80179322820112863823283078090753e-01 2.03285770626506802349897640169729e-02 1.15198070694543094547035444463745e-03
            5.00000000000000111022302462515654e-01 2.42110240962954900711956440773065e-02 1.38999135570895201996499679353292e-03
            4.19820677179887136176716921909247e-01 2.80934711299403137851893319520968e-02 1.59200206847138472573965906065041e-03
            3.41717950018184946081589714594884e-01 3.18753651485370395612939375951100e-02 1.73583406202048890626388200075780e-03
            2.67714629312019669615807515583583e-01 3.54587575101827437440427104320406e-02 1.80255971660516845292421272972661e-03
            1.99727347669159671639249609143008e-01 3.87508409383765656253117981577816e-02 1.77855764717937741767328496678147e-03
            1.39516911332385140642742271666066e-01 4.16663535883727981223856318138132e-02 1.65708994193382720509777339401580e-03
            8.86426717314284795534717886766884e-02 4.41297884700917394940056226460001e-02 1.43923524828972544849226000707176e-03
            4.84220481925909940201790959690697e-02 4.60773534414253907454117609177047e-02 1.13407844223793892977913966291226e-03
            1.98959239325848247226247167418478e-02 4.74586468050912438365607215473574e-02 7.58154067282971436299299217154157e-04
            3.79657807820771298068507348943967e-03 4.82382101059210879889782574991841e-02 3.34657173073832912065656008948622e-04
            9.96203421921792342530466157768387e-01 7.55364286481397320671682393822266e-05 8.27810621346007799167614985696995e-07
            9.80104076067414897721619126969017e-01 3.95847789131206025028258954634452e-04 9.98932022327091967525865279231567e-06
            9.51577951807409005979820904030930e-01 9.63401387499746931011856965199058e-04 3.74566065786464646026991676119877e-05
            9.11357328268571409424225748807658e-01 1.76362785394959141124782142640015e-03 9.08599479606107709832307772757076e-05
            8.60483088667614692823804034560453e-01 2.77581785517831995321880178551055e-03 1.74388219943065453767030992260345e-04
            8.00272652330840550405355315888301e-01 3.97376011648251786789742823202687e-03 2.88107162091609568868588731049840e-04
            7.32285370687980496917646178189898e-01 5.32642990043208364170368440682068e-03 4.27728271600950871698942901133478e-04
            6.58282049981815053918410285405116e-01 6.79879433996063119172159616709905e-03 5.84858135346190259086507357011442e-04
            5.80179322820112863823283078090753e-01 8.35272025849728358415369200429268e-03 7.47707531744118262563569832224175e-04
            5.00000000000000111022302462515654e-01 9.94796196629241062658888239411681e-03 9.02191329643538047697304271821395e-04
            4.19820677179887136176716921909247e-01 1.15432036740875411384710247375551e-02 1.03330891739027760228841756884322e-03
            3.41717950018184946081589714594884e-01 1.30971295926241935309031205747488e-02 1.12666487746325384364243227253155e-03
            2.67714629312019669615807515583583e-01 1.45694940321527393461975563582200e-02 1.16997400077818508243310269278936e-03
            1.99727347669159671639249609143008e-01 1.59221638161022999158333846025926e-02 1.15439515646342588314765542634177e-03
            1.39516911332385140642742271666066e-01 1.71201060774065078051719979157497e-02 1.07555501831858228538318922318240e-03
            8.86426717314284795534717886766884e-02 1.81322960786352363471429782748601e-02 9.34153695986177502567915365006002e-04
            4.84220481925909940201790959690697e-02 1.89325225450850780084532942737496e-02 7.36087842216016832036507011594040e-04
            1.98959239325848247226247167418478e-02 1.95000761434536221128333011165523e-02 4.92089409928605299622550450777680e-04
            3.79657807820771298068507348943967e-03 1.98203875039366858579192864908691e-02 2.17213437126905753362521300786625e-04
            9.96203421921792342530466157768387e-01 1.44140051039273714401185363942481e-05 3.59499118241754465868904129186023e-07
            9.80104076067414897721619126969017e-01 7.55364286481405858763790717169684e-05 4.33813207936528044347552351966968e-06
            9.51577951807409005979820904030930e-01 1.83838086669908389340563625857783e-04 1.62665429629989633593303149927678e-05
            9.11357328268571409424225748807658e-01 3.36538824289304560211794070667679e-04 3.94583861731804972023188171625918e-05
            8.60483088667614692823804034560453e-01 5.29686847103783301979329500852600e-04 7.57327940529975890448025066525872e-05
            8.00272652330840550405355315888301e-01 7.58280469779300920173115141409426e-04 1.25118315784179774197495960130766e-04
            7.32285370687980496917646178189898e-01 1.01639949286151730183103136795353e-03 1.85752553207866688642774244399902e-04
            6.58282049981815053918410285405116e-01 1.29735887796911990646475665300841e-03 2.53990440001360432645954867680871e-04
            5.80179322820112863823283078090753e-01 1.59388197975947657446249561274954e-03 3.24712188311457035198859255586967e-04
            5.00000000000000111022302462515654e-01 1.89828903910385605666166775051806e-03 3.91800949551530052449022090144126e-04
            4.19820677179887136176716921909247e-01 2.20269609844823662306301237379103e-03 4.48742303002993702006062637366313e-04
            3.41717950018184946081589714594884e-01 2.49921920023859307422031683643127e-03 4.89284649843478819213060670989535e-04
            2.67714629312019669615807515583583e-01 2.78017858534619546201360762438526e-03 5.08092806252761521046412696023253e-04
            1.99727347669159671639249609143008e-01 3.03829760842841108473000311107626e-03 5.01327272385516573525998662574921e-04
            1.39516911332385140642742271666066e-01 3.26689123110393032922704747988973e-03 4.67088813232813065100257343686962e-04
            8.86426717314284795534717886766884e-02 3.46003925391840912520469153434988e-03 4.05681470314135802189559054653500e-04
            4.84220481925909940201790959690697e-02 3.61273999153780478107989004854517e-03 3.19666024331580558588000728903467e-04
            1.98959239325848247226247167418478e-02 3.72104164955957324861790525005745e-03 2.13703115668886697029252297319601e-04
            3.79657807820771298068507348943967e-03 3.78216407310378579728626924350010e-03 9.43308011564449364711945622552491e-05
            ];
        
    case  37
        
        % ALG. DEG.:   37
        % PTS CARD.:  400
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.5e-15
        
        xyw=[
            9.96564299592547442330214835237712e-01 3.42389637016278806577740390082454e-03 2.66484302210678352018966659128796e-07
            9.81985963638956738108731769898441e-01 1.79521455289777601138645479750267e-02 3.22070197413265011892702198437188e-06
            9.56117214125662862045373913133517e-01 4.37320177690285263882863375783927e-02 1.21106166331857045842133532853602e-05
            9.19558485911109335653179641667521e-01 8.01651411461593577101325536204968e-02 2.94986497268677303081312740529185e-05
            8.73165953230075397861753572215093e-01 1.26398282983758325492829044378595e-01 5.69293939779793196085191930855274e-05
            8.18026840363257568000676656083670e-01 1.81347954378032999445125028614711e-01 9.47115399055740737064082446217128e-05
            7.55433500975413618760967437992804e-01 2.43726241804238352717248972112429e-01 1.41821949458491357396600074380899e-04
            6.86853044357709663358946272637695e-01 3.12071076519197576626396539722919e-01 1.95942460575123560476140660568944e-04
            6.13892925570822534098169853677973e-01 3.84780526196240812986104629089823e-01 2.53627371126371905889207436501920e-04
            5.38263260566748669155856532597682e-01 4.60150350329444679520207728273817e-01 3.10587131371550763045408238838263e-04
            4.61736739433251219821841004886664e-01 5.36413949263102818321158338221721e-01 3.62062681491019612480680711641412e-04
            3.86107074429177521412981377579854e-01 6.11783773396306518321807743632235e-01 4.03256141047899762525075217922677e-04
            3.13146955642290225618751264846651e-01 6.84493223073349921214969526772620e-01 4.29778010419811461444855593683201e-04
            2.44566499024586769817091180811985e-01 7.52838057788308590012604781804839e-01 4.38069204988765297740377357982311e-04
            1.81973159636742376488172112658503e-01 8.15216345214514359618362959736260e-01 4.25758292539158843433011458756710e-04
            1.26834046769924602138246427784907e-01 8.70166016608789116837385790859116e-01 3.91920070560901659915964412306266e-04
            8.04415140888906643468203583324794e-02 9.16399158446388084620082281617215e-01 3.37210630437482551408773234769001e-04
            4.38827858743368603988699305773480e-02 9.52832281823519089414276095340028e-01 2.63865860062384113317313083513227e-04
            1.80140363610431508689657675859053e-02 9.78612154063569827933122269314481e-01 1.75567766616818455574153268194948e-04
            3.43570040745255766978516476228833e-03 9.93140403222384615666840090852929e-01 7.72968275723136830166609678727241e-05
            9.96564299592547442330214835237712e-01 3.37380957538705632606235162995745e-03 6.14263614385752492261108873017328e-07
            9.81985963638956738108731769898441e-01 1.76895308550262739255565236362600e-02 7.42392710969514994161048970777728e-06
            9.56117214125662862045373913133517e-01 4.30922797739729523947005418449407e-02 2.79157574529835474010791562804457e-05
            9.19558485911109335653179641667521e-01 7.89924377291560120273317124883761e-02 6.79963024103375391227893631551638e-05
            8.73165953230075397861753572215093e-01 1.24549253639592918951883859790541e-01 1.31225948469030668933413275034638e-04
            8.18026840363257568000676656083670e-01 1.78695088522312167667038806939672e-01 2.18316247312923722477887022996867e-04
            7.55433500975413618760967437992804e-01 2.40160869218464440022486883208330e-01 3.26908799321071511737729631263960e-04
            6.86853044357709663358946272637695e-01 3.07505914997000140420624347825651e-01 4.51660090467010777014883471380813e-04
            6.13892925570822534098169853677973e-01 3.79151727551154227313645606045611e-01 5.84627553678841054875248151034839e-04
            5.38263260566748669155856532597682e-01 4.53418997019871206077823444502428e-01 7.15923498364874578633876289757154e-04
            4.61736739433251219821841004886664e-01 5.28566966619085643053210787911667e-01 8.34577982725015054404904546458965e-04
            3.86107074429177521412981377579854e-01 6.02834236087802399772783701337175e-01 9.29531580916543318307554422119665e-04
            3.13146955642290225618751264846651e-01 6.74480048641956764221561115846271e-01 9.90666211382609188801362520848670e-04
            2.44566499024586769817091180811985e-01 7.41825094420491826241459420998581e-01 1.00977795305463705945181462908522e-03
            1.81973159636742376488172112658503e-01 8.03290875116644542686117347329855e-01 9.81400500743381332471870770461919e-04
            1.26834046769924602138246427784907e-01 8.57436709999363833034635717922356e-01 9.03401202607167336765259246789128e-04
            8.04415140888906643468203583324794e-02 9.02993525909800753836975673038978e-01 7.77292391872552485172775948996104e-04
            4.38827858743368603988699305773480e-02 9.38893683864983952247484921826981e-01 6.08227935267964883725433367089863e-04
            1.80140363610431508689657675859053e-02 9.64296432783930623777735036128433e-01 4.04695098349247854227495047041430e-04
            3.43570040745255766978516476228833e-03 9.78612154063569716910819806798827e-01 1.78174204976562733327913323755354e-04
            9.96564299592547442330214835237712e-01 3.28493230214394436872193949739085e-03 9.48172493430292145066547258164258e-07
            9.81985963638956738108731769898441e-01 1.72235302606790763313426850800170e-02 1.14595156115235156154413820317473e-05
            9.56117214125662862045373913133517e-01 4.19570869782442124051868859169190e-02 4.30905440763554433911131091861080e-05
            9.19558485911109335653179641667521e-01 7.69115163507203963844105487623892e-02 1.04958558655505679118279116490697e-04
            8.73165953230075397861753572215093e-01 1.21268215453944339099479066135245e-01 2.02559343983050996933562015733798e-04
            8.18026840363257568000676656083670e-01 1.73987670437526636968428306317946e-01 3.36991245652789044202901624913693e-04
            7.55433500975413618760967437992804e-01 2.33834239715854180774456949620799e-01 5.04613856522368006869705414629834e-04
            6.86853044357709663358946272637695e-01 2.99405194840639155007266936081578e-01 6.97178970285090675307326790743900e-04
            6.13892925570822534098169853677973e-01 3.69163620357435107166566012892872e-01 9.02426502754899244850006478912974e-04
            5.38263260566748669155856532597682e-01 4.41474444966387380429040376839112e-01 1.10509389234907215611436459568040e-03
            4.61736739433251219821841004886664e-01 5.14642769159275648149787230067886e-01 1.28824802301484835770351011774437e-03
            3.86107074429177521412981377579854e-01 5.86953593768227643856505437724991e-01 1.43481765183368936179830743782304e-03
            3.13146955642290225618751264846651e-01 6.56712019285023762549258208309766e-01 1.52918458753753018497523186169929e-03
            2.44566499024586769817091180811985e-01 7.22282974409808264937282729079016e-01 1.55868532196257506515502111454907e-03
            1.81973159636742376488172112658503e-01 7.82129543688136141810218759928830e-01 1.51488210932711957028284910364846e-03
            1.26834046769924602138246427784907e-01 8.34848998671718578457046078256099e-01 1.39448300498885976658747232193036e-03
            8.04415140888906643468203583324794e-02 8.79205697774942507294326787814498e-01 1.19982243464506962982163429387583e-03
            4.38827858743368603988699305773480e-02 9.14160127147418899440367567876820e-01 9.38855866521858765244568090224675e-04
            1.80140363610431508689657675859053e-02 9.38893683864983841225182459311327e-01 6.24684177109422855526943418880137e-04
            3.43570040745255766978516476228833e-03 9.52832281823518867369671170308720e-01 2.75028304202134588930805936257684e-04
            9.96564299592547442330214835237712e-01 3.15932746472125536990604999232346e-03 1.25990322327730073706649957521186e-06
            9.81985963638956738108731769898441e-01 1.65649600013086130412265362110702e-02 1.52270612743908618662075013516777e-05
            9.56117214125662862045373913133517e-01 4.03527881361668769311101812036213e-02 5.72574249419144446762372502313099e-05
            9.19558485911109335653179641667521e-01 7.39706768999774666450974791587214e-02 1.39465790535857906079844625857334e-04
            8.73165953230075397861753572215093e-01 1.16631324009730696911191216713632e-01 2.69154792147472960838622269008624e-04
            8.18026840363257568000676656083670e-01 1.67334963152023424859393685437681e-01 4.47783878519985374840051139955222e-04
            7.55433500975413618760967437992804e-01 2.24893199547629457635267158366332e-01 6.70515785627629683045280994946324e-04
            6.86853044357709663358946272637695e-01 2.87956940398097827049639363394817e-01 9.26390543861428574640148880803281e-04
            6.13892925570822534098169853677973e-01 3.55048036761662411109341519477312e-01 1.19911732039223854241749567250963e-03
            5.38263260566748669155856532597682e-01 4.24593937002772980893183785155998e-01 1.46841568031314591985525552786385e-03
            4.61736739433251219821841004886664e-01 4.94964548908336410271147087769350e-01 1.71178540594981148242892388111613e-03
            3.86107074429177521412981377579854e-01 5.64510449149446813521535659674555e-01 1.90654274078383596802144239745758e-03
            3.13146955642290225618751264846651e-01 6.31601545513011619625842740788357e-01 2.03193469982911436091943535586779e-03
            2.44566499024586769817091180811985e-01 6.94665286363479461684278248867486e-01 2.07113439255242424291703073890858e-03
            1.81973159636742376488172112658503e-01 7.52223522759085883038210340600926e-01 2.01292999496475175438825466756043e-03
            1.26834046769924602138246427784907e-01 8.02927161901378694253139656211715e-01 1.85294726957827070139284142413771e-03
            8.04415140888906643468203583324794e-02 8.45587809011131841252506546879886e-01 1.59428813137245492699556326243737e-03
            4.38827858743368603988699305773480e-02 8.79205697774942618316629250330152e-01 1.24752356835866715263627657606094e-03
            1.80140363610431508689657675859053e-02 9.02993525909800864859278135554632e-01 8.30061633008500536316798257274741e-04
            3.43570040745255766978516476228833e-03 9.16399158446388084620082281617215e-01 3.65449376941068206255114336755696e-04
            9.96564299592547442330214835237712e-01 2.99993662128627104970779448933627e-03 1.54211228820107622252604060580916e-06
            9.81985963638956738108731769898441e-01 1.57292432307115787049056621071941e-02 1.86378111196089936939506576107917e-05
            9.56117214125662862045373913133517e-01 3.83169545583568749158587252168218e-02 7.00826674321818803101852290993179e-05
            9.19558485911109335653179641667521e-01 7.02387913286967591197651472612051e-02 1.70705102896373143517641524091744e-04
            8.73165953230075397861753572215093e-01 1.10747171349889178548764334664156e-01 3.29443487984052590412942240760685e-04
            8.18026840363257568000676656083670e-01 1.58892767396504847043914310233959e-01 5.48084177233685199311097324681441e-04
            7.55433500975413618760967437992804e-01 2.13547140248945277463121783512179e-01 8.20706395019344856454901471209951e-04
            6.86853044357709663358946272637695e-01 2.73429260024496578740382801697706e-01 1.13389521906759739404424447428710e-03
            6.13892925570822534098169853677973e-01 3.37135551692828439040994226161274e-01 1.46771078969183720389080072266097e-03
            5.38263260566748669155856532597682e-01 4.03172800228581817982842494529905e-01 1.79733000357571478904084028016541e-03
            4.61736739433251219821841004886664e-01 4.69993153001493635390062308943016e-01 2.09521275960532086038434762542693e-03
            3.86107074429177521412981377579854e-01 5.36030401537246903309608114795992e-01 2.33359430647016048016029188261200e-03
            3.13146955642290225618751264846651e-01 5.99736693205578874632522001775214e-01 2.48707314302899860219975280983817e-03
            2.44566499024586769817091180811985e-01 6.59618812981129787331724401155952e-01 2.53505327890409925400971857811783e-03
            1.81973159636742376488172112658503e-01 7.14273185833570467551112415094394e-01 2.46381152391135461662829975182376e-03
            1.26834046769924602138246427784907e-01 7.62418781880186191557413621922024e-01 2.26799384350520605302392596058780e-03
            8.04415140888906643468203583324794e-02 8.02927161901378694253139656211715e-01 1.95139695882932950876664790484938e-03
            4.38827858743368603988699305773480e-02 8.34848998671718689479348540771753e-01 1.52695968153970578970457516732040e-03
            1.80140363610431508689657675859053e-02 8.57436709999363944056938180438010e-01 1.01598933995657061783723928982681e-03
            3.43570040745255766978516476228833e-03 8.70166016608789116837385790859116e-01 4.47307352250665409085922386012157e-04
            9.96564299592547442330214835237712e-01 2.81049514874317238680156982866265e-03 1.78817841539713588027609228459092e-06
            9.81985963638956738108731769898441e-01 1.47359652466130530701349243827281e-02 2.16117411224388875740868437569375e-05
            9.56117214125662862045373913133517e-01 3.58972966751213984348822805259260e-02 8.12653619029737639455415743050537e-05
            9.19558485911109335653179641667521e-01 6.58033176041716988402541232971998e-02 1.97943549722651471970552572443580e-04
            8.73165953230075397861753572215093e-01 1.03753654529687058816200817545905e-01 3.82010920225165338543621418310181e-04
            8.18026840363257568000676656083670e-01 1.48858928808563051315871916813194e-01 6.35538866429279395917073536281805e-04
            7.55433500975413618760967437992804e-01 2.00061960455786114199128178370302e-01 9.51661868062769365425590528673183e-04
            6.86853044357709663358946272637695e-01 2.56162614693435930490750251919962e-01 1.31482445965329375336949624397676e-03
            6.13892925570822534098169853677973e-01 3.15845950137201181284751783095999e-01 1.70190509099309877395767021113215e-03
            5.38263260566748669155856532597682e-01 3.77713046038215316535513466078555e-01 2.08411977670505633650210874918685e-03
            4.61736739433251219821841004886664e-01 4.40313794325042306976314421262941e-01 2.42953399765814034155786416135925e-03
            3.86107074429177521412981377579854e-01 5.02180890226056275693622410472017e-01 2.70595273836472241177353303953623e-03
            3.13146955642290225618751264846651e-01 5.61864225669821748532228866679361e-01 2.88392132395645693979102652804158e-03
            2.44566499024586769817091180811985e-01 6.17964879907471065223489858908579e-01 2.93955737847474479415565973283719e-03
            1.81973159636742376488172112658503e-01 6.69167911554694461173653508012649e-01 2.85694797997131620659327921885051e-03
            1.26834046769924602138246427784907e-01 7.14273185833570467551112415094394e-01 2.62988478091991714075126296279450e-03
            8.04415140888906643468203583324794e-02 7.52223522759085883038210340600926e-01 2.26277032376207302630244377894542e-03
            4.38827858743368603988699305773480e-02 7.82129543688136363854823684960138e-01 1.77060799307693426778154144329847e-03
            1.80140363610431508689657675859053e-02 8.03290875116644653708419809845509e-01 1.17810500693386194237533004525176e-03
            3.43570040745255766978516476228833e-03 8.15216345214514359618362959736260e-01 5.18681654029326359711671301511160e-04
            9.96564299592547442330214835237712e-01 2.59544318710454085738503771096930e-03 1.99233227561257533345787963663387e-06
            9.81985963638956738108731769898441e-01 1.36084065549213120011051714186578e-02 2.40791237606208626464629474206447e-05
            9.56117214125662862045373913133517e-01 3.31505265656049305511565705728572e-02 9.05433160441505570066936248174727e-05
            9.19558485911109335653179641667521e-01 6.07682146119337338041610507843870e-02 2.20542491434880586192307516668620e-04
            8.73165953230075397861753572215093e-01 9.58146879942834983623356492898893e-02 4.25624579431033362494213312388069e-04
            8.18026840363257568000676656083670e-01 1.37468621067942109448267729021609e-01 7.08097461131736074374520839569414e-04
            7.55433500975413618760967437992804e-01 1.84753726579443361410781676568149e-01 1.06031178929655279206956119963934e-03
            6.86853044357709663358946272637695e-01 2.36561701020647940918451013203594e-01 1.46493615244222877801172355560766e-03
            6.13892925570822534098169853677973e-01 2.91678218987408144347028837728431e-01 1.89620924490460011739567658395345e-03
            5.38263260566748669155856532597682e-01 3.48811401599033399367755237108213e-01 2.32206085344665311490097714397507e-03
            4.61736739433251219821841004886664e-01 4.06622099376380330415514663400245e-01 2.70691053898967264393360387941811e-03
            3.86107074429177521412981377579854e-01 4.63755281988005418902787369006546e-01 3.01488762558905316543689245634141e-03
            3.13146955642290225618751264846651e-01 5.18871799954765733353667656047037e-01 3.21317463882357179719151396568577e-03
            2.44566499024586769817091180811985e-01 5.70679774395969952038853989506606e-01 3.27516258485294371738394403337225e-03
            1.81973159636742376488172112658503e-01 6.17964879907471509312699708971195e-01 3.18312178540577613364281894803298e-03
            1.26834046769924602138246427784907e-01 6.59618812981130120398631788702914e-01 2.93013509449245633867175975240116e-03
            8.04415140888906643468203583324794e-02 6.94665286363479905773488098930102e-01 2.52110768674515756462839455309677e-03
            4.38827858743368603988699305773480e-02 7.22282974409808820048795041657286e-01 1.97275586244079073963697545934792e-03
            1.80140363610431508689657675859053e-02 7.41825094420492381352971733576851e-01 1.31260762861508221976958310506234e-03
            3.43570040745255766978516476228833e-03 7.52838057788309034101814631867455e-01 5.77898822171633158584214129405154e-04
            9.96564299592547442330214835237712e-01 2.35982128435981283395839191996401e-03 2.14978807714870044011270903971944e-06
            9.81985963638956738108731769898441e-01 1.23729957157530431194336628664132e-02 2.59821184460080637705463313391618e-05
            9.56117214125662862045373913133517e-01 3.01410250726859597902862475393704e-02 9.76990352863572924157695154256942e-05
            9.19558485911109335653179641667521e-01 5.52514988446981478165120904577634e-02 2.37972161769853295457122221989721e-04
            8.73165953230075397861753572215093e-01 8.71163511521308442375755021203076e-02 4.59262070590576076869332444729821e-04
            8.18026840363257568000676656083670e-01 1.24988818687887998093444252845075e-01 7.64059036754914742278765871219548e-04
            7.55433500975413618760967437992804e-01 1.67981244402943985516429847848485e-01 1.14410917826901210696266542043986e-03
            6.86853044357709663358946272637695e-01 2.15085939814255777680074288582546e-01 1.58071136670016052297449427044285e-03
            6.13892925570822534098169853677973e-01 2.65198819519729345195457881345646e-01 2.04606835735855318442188988115049e-03
            5.38263260566748669155856532597682e-01 3.17145285171531232659702936871327e-01 2.50557539937383035033580114259166e-03
            4.61736739433251219821841004886664e-01 3.69707759186178541721545798282023e-01 2.92084009974637702489186352750039e-03
            3.86107074429177521412981377579854e-01 4.21654224837980262652337160034222e-01 3.25315689093160897299439326957327e-03
            3.13146955642290225618751264846651e-01 4.71767104543453941190023215312976e-01 3.46711470415527345939610270875164e-03
            2.44566499024586769817091180811985e-01 5.18871799954765400286760268500075e-01 3.53400161299677589149537126900213e-03
            1.81973159636742376488172112658503e-01 5.61864225669821637509926404163707e-01 3.43468674685482250677481808054381e-03
            1.26834046769924602138246427784907e-01 5.99736693205578763610219539259560e-01 3.16170629150605954696406918458251e-03
            8.04415140888906643468203583324794e-02 6.31601545513011508603540278272703e-01 2.72035308192066470345249484807937e-03
            4.38827858743368603988699305773480e-02 6.56712019285023873571560670825420e-01 2.12866452253625531437775109111499e-03
            1.80140363610431508689657675859053e-02 6.74480048641956653199258653330617e-01 1.41634418340360213744988371331601e-03
            3.43570040745255766978516476228833e-03 6.84493223073349810192667064256966e-01 6.23570682917772944713064031674321e-04
            9.96564299592547442330214835237712e-01 2.10915217451591776448571735613768e-03 2.25685495042602009014426345245852e-06
            9.81985963638956738108731769898441e-01 1.10586894830200215683602138483366e-02 2.72761177070071468434356298793375e-05
            9.56117214125662862045373913133517e-01 2.69393318025947896277028092981709e-02 1.02564784771857513329985922556631e-04
            9.19558485911109335653179641667521e-01 4.93824764213756303710134432094492e-02 2.49823997566401103439387432203489e-04
            8.73165953230075397861753572215093e-01 7.78625240335755475218348919952405e-02 4.82134908353365238160848083026622e-04
            8.18026840363257568000676656083670e-01 1.11712035344766091871093749432475e-01 8.02111816437798327145136667581937e-04
            7.55433500975413618760967437992804e-01 1.50137643582817059684231253413600e-01 1.20108976798723936056656480531046e-03
            6.86853044357709663358946272637695e-01 1.92238700732842215934681462385925e-01 1.65943625376483014019490802581913e-03
            6.13892925570822534098169853677973e-01 2.37028401504919078446320668263070e-01 2.14796963026197022458840812930703e-03
            5.38263260566748669155856532597682e-01 2.83456917814211228900944661290850e-01 2.63036171046330787340328249968024e-03
            4.61736739433251219821841004886664e-01 3.30436007756611360708376423644950e-01 3.06630802756074644782602334203148e-03
            3.86107074429177521412981377579854e-01 3.76864524065903372385122338528163e-01 3.41517534302693227463154634904186e-03
            3.13146955642290225618751264846651e-01 4.21654224837980373674639622549876e-01 3.63978899452536020647785974801991e-03
            2.44566499024586769817091180811985e-01 4.63755281988005196858182443975238e-01 3.71000710250642736637294305523938e-03
            1.81973159636742376488172112658503e-01 5.02180890226056386715924872987671e-01 3.60574601292003218272474995842458e-03
            1.26834046769924602138246427784907e-01 5.36030401537247014331910577311646e-01 3.31917018780869413699385894744864e-03
            8.04415140888906643468203583324794e-02 5.64510449149446924543838122190209e-01 2.85583606360966402865075863815036e-03
            4.38827858743368603988699305773480e-02 5.86953593768227865901110362756299e-01 2.23467936981673522683911059516504e-03
            1.80140363610431508689657675859053e-02 6.02834236087802621817388626368484e-01 1.48688301688837507148754912833510e-03
            3.43570040745255766978516476228833e-03 6.11783773396306629344110206147889e-01 6.54626657223835090905483813372712e-04
            9.96564299592547442330214835237712e-01 1.84931130364592065731810244244571e-03 2.31102323612482596473837251349615e-06
            9.81985963638956738108731769898441e-01 9.69629394766311457354301950317677e-03 2.79307900582047799887663169782215e-05
            9.56117214125662862045373913133517e-01 2.36204914074731701922882365352052e-02 1.05026510795991137050230113114679e-04
            9.19558485911109335653179641667521e-01 4.32987116584123421514362917150720e-02 2.55820190486127820982814595751620e-04
            8.73165953230075397861753572215093e-01 6.82701075652551170325210705414065e-02 4.93706950879236358513402205261400e-04
            8.18026840363257568000676656083670e-01 9.79494662417064176906933425925672e-02 8.21363838827180977111841464477493e-04
            7.55433500975413618760967437992804e-01 1.31641161190368422007068716084177e-01 1.22991792714295471687646621461454e-03
            6.86853044357709663358946272637695e-01 1.68555501380570210701037581202399e-01 1.69926549359963722542199437981481e-03
            6.13892925570822534098169853677973e-01 2.07827252810137363958631340210559e-01 2.19952448653770249800687430763446e-03
            5.38263260566748669155856532597682e-01 2.48535922890801092011159312278323e-01 2.69349478181850560468135746816642e-03
            4.61736739433251219821841004886664e-01 2.89727337675947604900272835948272e-01 3.13990453815886838215276632979567e-03
            3.86107074429177521412981377579854e-01 3.30436007756611249686073961129296e-01 3.49714525148634792009461058626130e-03
            3.13146955642290225618751264846651e-01 3.69707759186178541721545798282023e-01 3.72715000552059552013650645108100e-03
            2.44566499024586769817091180811985e-01 4.06622099376379997348607275853283e-01 3.79905346529338079977766717831855e-03
            1.81973159636742376488172112658503e-01 4.40313794325042251465163190005114e-01 3.69228993553602155563986286779254e-03
            1.26834046769924602138246427784907e-01 4.69993153001493579878911077685189e-01 3.39883581230185968097279314292791e-03
            8.04415140888906643468203583324794e-02 4.94964548908336354759995856511523e-01 2.92438089577681760086735884840436e-03
            4.38827858743368603988699305773480e-02 5.14642769159275648149787230067886e-01 2.28831540456793207372054865800237e-03
            1.80140363610431508689657675859053e-02 5.28566966619085643053210787911667e-01 1.52257069103168152929272949336337e-03
            3.43570040745255766978516476228833e-03 5.36413949263102707298855875706067e-01 6.70338789626434257765130730888359e-04
            9.96564299592547442330214835237712e-01 1.58638910380663657878619332564085e-03 2.31102323612481410627711095329140e-06
            9.81985963638956738108731769898441e-01 8.31774241338014558300173462157545e-03 2.79307900582046342990993892385632e-05
            9.56117214125662862045373913133517e-01 2.02622944668639642928908983776637e-02 1.05026510795990581396616714293657e-04
            9.19558485911109335653179641667521e-01 3.71428024304783152564901627101790e-02 2.55820190486126465730098988871077e-04
            8.73165953230075397861753572215093e-01 5.85639392046694712279375494290434e-02 4.93706950879233756428188240050758e-04
            8.18026840363257568000676656083670e-01 8.40236933950359449196909622514795e-02 8.21363838827176640303151522459757e-04
            7.55433500975413618760967437992804e-01 1.12925337834217931476388230294106e-01 1.22991792714294821166343130158793e-03
            6.86853044357709663358946272637695e-01 1.44591454261720098184440530530992e-01 1.69926549359962811812374550157756e-03
            6.13892925570822534098169853677973e-01 1.78279821619040046432047574853641e-01 2.19952448653769078862341146418657e-03
            5.38263260566748669155856532597682e-01 2.13200816542450183321832923866168e-01 2.69349478181849129321268065950790e-03
            4.61736739433251219821841004886664e-01 2.48535922890801092011159312278323e-01 3.13990453815885190227974455012827e-03
            3.86107074429177521412981377579854e-01 2.83456917814211117878642198775196e-01 3.49714525148632927181724383558503e-03
            3.13146955642290225618751264846651e-01 3.17145285171531177148551705613500e-01 3.72715000552057600449740171200119e-03
            2.44566499024586769817091180811985e-01 3.48811401599033066300847849561251e-01 3.79905346529336085045769344503697e-03
            1.81973159636742376488172112658503e-01 3.77713046038215261024362234820728e-01 3.69228993553600247368162712291451e-03
            1.26834046769924602138246427784907e-01 4.03172800228581762471691263272078e-01 3.39883581230184146637629538645342e-03
            8.04415140888906643468203583324794e-02 4.24593937002772925382032553898171e-01 2.92438089577680242203694405134229e-03
            4.38827858743368603988699305773480e-02 4.41474444966387380429040376839112e-01 2.28831540456791993065621682035271e-03
            1.80140363610431508689657675859053e-02 4.53418997019871150566672213244601e-01 1.52257069103167350619665310063056e-03
            3.43570040745255766978516476228833e-03 4.60150350329444568497905265758163e-01 6.70338789626430679897961528723727e-04
            9.96564299592547442330214835237712e-01 1.32654823293664012213988190325153e-03 2.25685495042600823168300189225377e-06
            9.81985963638956738108731769898441e-01 6.95534687802324119026975424162629e-03 2.72761177070069977656369131224778e-05
            9.56117214125662862045373913133517e-01 1.69434540717423483269232775683122e-02 1.02564784771856957676372523735608e-04
            9.19558485911109335653179641667521e-01 3.10590376675150409147008190302586e-02 2.49823997566399748186671825322946e-04
            8.73165953230075397861753572215093e-01 4.89715227363490615553054396968946e-02 4.82134908353362636075634117815980e-04
            8.18026840363257568000676656083670e-01 7.02611242919762846170783632260282e-02 8.02111816437793990336446725564201e-04
            7.55433500975413618760967437992804e-01 9.44288554417693493103769242225098e-02 1.20108976798723285535352989228386e-03
            6.86853044357709663358946272637695e-01 1.20908254909448148461947880605294e-01 1.65943625376482124973709364468277e-03
            6.13892925570822534098169853677973e-01 1.49078672924258415211085093687871e-01 2.14796963026195851520494528585914e-03
            5.38263260566748669155856532597682e-01 1.78279821619040129698774421740382e-01 2.63036171046329399561547468522349e-03
            4.61736739433251219821841004886664e-01 2.07827252810137447225358187097299e-01 3.06630802756072996795300156236408e-03
            3.86107074429177521412981377579854e-01 2.37028401504919078446320668263070e-01 3.41517534302691362635417959836559e-03
            3.13146955642290225618751264846651e-01 2.65198819519729400706609112603473e-01 3.63978899452534025715788601473832e-03
            2.44566499024586769817091180811985e-01 2.91678218987407977813575143954949e-01 3.71000710250640741705296932195779e-03
            1.81973159636742376488172112658503e-01 3.15845950137201181284751783095999e-01 3.60574601292001266708564521934477e-03
            1.26834046769924602138246427784907e-01 3.37135551692828439040994226161274e-01 3.31917018780867635607823018517593e-03
            8.04415140888906643468203583324794e-02 3.55048036761662466620492750735139e-01 2.85583606360964841613947484688651e-03
            4.38827858743368603988699305773480e-02 3.69163620357435273700019706666353e-01 2.23467936981672308377477875751538e-03
            1.80140363610431508689657675859053e-02 3.79151727551154338335948068561265e-01 1.48688301688836704839147273560229e-03
            3.43570040745255766978516476228833e-03 3.84780526196240868497255860347650e-01 6.54626657223831513038314611208079e-04
            9.96564299592547442330214835237712e-01 1.07587912309274461898633834522343e-03 2.14978807714870594582686619267164e-06
            9.81985963638956738108731769898441e-01 5.64104064529021790447282924674255e-03 2.59821184460081315331821116831890e-05
            9.56117214125662862045373913133517e-01 1.37417608016511712254459354198843e-02 9.76990352863575499137854807329973e-05
            9.19558485911109335653179641667521e-01 2.51900152441925095914143639674876e-02 2.37972161769853891768317089017160e-04
            8.73165953230075397861753572215093e-01 3.97176956177937440228831178501423e-02 4.59262070590577269491722178784698e-04
            8.18026840363257568000676656083670e-01 5.69843409488543645169400519989722e-02 7.64059036754916693842676345127529e-04
            7.55433500975413618760967437992804e-01 7.65852546216423679670270985297975e-02 1.14410917826901514272874837985228e-03
            6.86853044357709663358946272637695e-01 9.80610158280345173276160153363890e-02 1.58071136670016442610231521825881e-03
            6.13892925570822534098169853677973e-01 1.20908254909448092950796649347467e-01 2.04606835735855838859231781157177e-03
            5.38263260566748669155856532597682e-01 1.44591454261720070428864914902078e-01 2.50557539937383685554883605561827e-03
            4.61736739433251219821841004886664e-01 1.68555501380570182945461965573486e-01 2.92084009974638439746663642893054e-03
            3.86107074429177521412981377579854e-01 1.92238700732842104912378999870270e-01 3.25315689093161721293090415940696e-03
            3.13146955642290225618751264846651e-01 2.15085939814255749924498672953632e-01 3.46711470415528256669435158698889e-03
            2.44566499024586769817091180811985e-01 2.36561701020647718873846088172286e-01 3.53400161299678499879362014723938e-03
            1.81973159636742376488172112658503e-01 2.56162614693435819468447789404308e-01 3.43468674685483118039219796457928e-03
            1.26834046769924602138246427784907e-01 2.73429260024496467718080339182052e-01 3.16170629150606735321971108021444e-03
            8.04415140888906643468203583324794e-02 2.87956940398097716027336900879163e-01 2.72035308192067164234639875530775e-03
            4.38827858743368603988699305773480e-02 2.99405194840639099496115704823751e-01 2.12866452253626095222904801573804e-03
            1.80140363610431508689657675859053e-02 3.07505914997000029398321885309997e-01 1.41634418340360560689683566693020e-03
            3.43570040745255766978516476228833e-03 3.12071076519197465604094077207264e-01 6.23570682917774462596105511380529e-04
            9.96564299592547442330214835237712e-01 8.40257220348018330283168531025240e-04 1.99233227561257448642493238233353e-06
            9.81985963638956738108731769898441e-01 4.40562980612195769641870057853339e-03 2.40791237606208524820675803690406e-05
            9.56117214125662862045373913133517e-01 1.07322593087322230159808000848898e-02 9.05433160441505163491121566110564e-05
            9.19558485911109335653179641667521e-01 1.96732994769569617676818751306200e-02 2.20542491434880477772090268118177e-04
            8.73165953230075397861753572215093e-01 3.10193587756411592870620097528445e-02 4.25624579431033199863887439562404e-04
            8.18026840363257568000676656083670e-01 4.45045385688003294899495188019500e-02 7.08097461131735749113869093918083e-04
            7.55433500975413618760967437992804e-01 5.98127724451431030949777323257877e-02 1.06031178929655235838869220543756e-03
            6.86853044357709663358946272637695e-01 7.65852546216425206226929844888218e-02 1.46493615244222812749042006430500e-03
            6.13892925570822534098169853677973e-01 9.44288554417694880882550023670774e-02 1.89620924490459925003393859554990e-03
            5.38263260566748669155856532597682e-01 1.12925337834218139643205347510957e-01 2.32206085344665224753923915557152e-03
            4.61736739433251219821841004886664e-01 1.31641161190368671807249256744399e-01 2.70691053898967134289099689681279e-03
            3.86107074429177521412981377579854e-01 1.50137643582817253973260562815994e-01 3.01488762558905186439428547373609e-03
            3.13146955642290225618751264846651e-01 1.67981244402944290827761619766534e-01 3.21317463882357049614890698308045e-03
            2.44566499024586769817091180811985e-01 1.84753726579443555699810985970544e-01 3.27516258485294241634133705076692e-03
            1.81973159636742376488172112658503e-01 2.00061960455786419510459950288350e-01 3.18312178540577483260021196542766e-03
            1.26834046769924602138246427784907e-01 2.13547140248945610530029171059141e-01 2.93013509449245503762915276979584e-03
            8.04415140888906643468203583324794e-02 2.24893199547629818457750161542208e-01 2.52110768674515626358578757049145e-03
            4.38827858743368603988699305773480e-02 2.33834239715854597108091184054501e-01 1.97275586244078987227523747094438e-03
            1.80140363610431508689657675859053e-02 2.40160869218464828600545502013119e-01 1.31260762861508156924827961375968e-03
            3.43570040745255766978516476228833e-03 2.43726241804238741295307590917218e-01 5.77898822171632941743779632304268e-04
            9.96564299592547442330214835237712e-01 6.25205258709385066143160436524795e-04 1.78817841539713312741901370811481e-06
            9.81985963638956738108731769898441e-01 3.27807111443020708640982974202416e-03 2.16117411224388536927689535849240e-05
            9.56117214125662862045373913133517e-01 7.98548919921573605029685438694287e-03 8.12653619029736419727971696858049e-05
            9.19558485911109335653179641667521e-01 1.46381964847189620371192830816653e-02 1.97943549722651173814955138929861e-04
            8.73165953230075397861753572215093e-01 2.30803922402375398525986582853875e-02 3.82010920225164796442535175557964e-04
            8.18026840363257568000676656083670e-01 3.31142308281793251723001958453096e-02 6.35538866429278420135118299327814e-04
            7.55433500975413618760967437992804e-01 4.45045385688002531621165758224379e-02 9.51661868062767955962766297517419e-04
            6.86853044357709663358946272637695e-01 5.69843409488543853336217637206573e-02 1.31482445965329180180558577006877e-03
            6.13892925570822534098169853677973e-01 7.02611242919762846170783632260282e-02 1.70190509099309617187245624592151e-03
            5.38263260566748669155856532597682e-01 8.40236933950359587974787700659363e-02 2.08411977670505286705515679557266e-03
            4.61736739433251219821841004886664e-01 9.79494662417064315684811504070240e-02 2.42953399765813643843004321354329e-03
            3.86107074429177521412981377579854e-01 1.11712035344766077993305941618019e-01 2.70595273836471807496484309751850e-03
            3.13146955642290225618751264846651e-01 1.24988818687888011971232060659531e-01 2.88392132395645260298233658602385e-03
            2.44566499024586769817091180811985e-01 1.37468621067942026181540882134868e-01 2.93955737847474045734696979081946e-03
            1.81973159636742376488172112658503e-01 1.48858928808563051315871916813194e-01 2.85694797997131186978458927683278e-03
            1.26834046769924602138246427784907e-01 1.58892767396504847043914310233959e-01 2.62988478091991280394257302077676e-03
            8.04415140888906643468203583324794e-02 1.67334963152023424859393685437681e-01 2.26277032376206955685549182533123e-03
            4.38827858743368603988699305773480e-02 1.73987670437526692479579537575773e-01 1.77060799307693144885589298098694e-03
            1.80140363610431508689657675859053e-02 1.78695088522312195422614422568586e-01 1.17810500693386020765185406844466e-03
            3.43570040745255766978516476228833e-03 1.81347954378032999445125028614711e-01 5.18681654029325492349933313107613e-04
            9.96564299592547442330214835237712e-01 4.35763786166286728497587521502510e-04 1.54211228820109888065737965834323e-06
            9.81985963638956738108731769898441e-01 2.28479313033168362004343698856701e-03 1.86378111196092647444937789869002e-05
            9.56117214125662862045373913133517e-01 5.56583131598026303876736164966132e-03 7.00826674321829103022490903285302e-05
            9.19558485911109335653179641667521e-01 1.02027227601939069617786870480813e-02 1.70705102896375664287692552889553e-04
            8.73165953230075397861753572215093e-01 1.60868754200354201200351411671363e-02 3.29443487984057415112609801255417e-04
            8.18026840363257568000676656083670e-01 2.30803922402375398525986582853875e-02 5.48084177233693222407173717414253e-04
            7.55433500975413618760967437992804e-01 3.10193587756411072453577304486316e-02 8.20706395019356891099016060309168e-04
            6.86853044357709663358946272637695e-01 3.97176956177937579006709256645991e-02 1.13389521906761409075770075105538e-03
            6.13892925570822534098169853677973e-01 4.89715227363490546164115357896662e-02 1.46771078969185867109381593564876e-03
            5.38263260566748669155856532597682e-01 5.85639392046694920446192611507286e-02 1.79733000357574124357384892647360e-03
            4.61736739433251219821841004886664e-01 6.82701075652551309103088783558633e-02 2.09521275960535165172604621375285e-03
            3.86107074429177521412981377579854e-01 7.78625240335755336440470841807837e-02 2.33359430647019474094894242455211e-03
            3.13146955642290225618751264846651e-01 8.71163511521308581153633099347644e-02 2.48707314302903546507361731698893e-03
            2.44566499024586769817091180811985e-01 9.58146879942834428511844180320622e-02 2.53505327890413655056445207947036e-03
            1.81973159636742376488172112658503e-01 1.03753654529687058816200817545905e-01 2.46381152391139104582129526477274e-03
            1.26834046769924602138246427784907e-01 1.10747171349889178548764334664156e-01 2.26799384350523944645083851412437e-03
            8.04415140888906643468203583324794e-02 1.16631324009730696911191216713632e-01 1.95139695882935813170400152216644e-03
            4.38827858743368603988699305773480e-02 1.21268215453944366855054681764159e-01 1.52695968153972812426932836871174e-03
            1.80140363610431508689657675859053e-02 1.24549253639592932829671667604998e-01 1.01598933995658557982721958978800e-03
            3.43570040745255766978516476228833e-03 1.26398282983758325492829044378595e-01 4.47307352250671968509065923313983e-04
            9.96564299592547442330214835237712e-01 2.76372942731302299879114769964872e-04 1.25990322327729163146231659148322e-06
            9.81985963638956738108731769898441e-01 1.44907635973465036792473537019532e-03 1.52270612743907534459902528012343e-05
            9.56117214125662862045373913133517e-01 3.52999773817026319192025063387064e-03 5.72574249419140313241589901327444e-05
            9.19558485911109335653179641667521e-01 6.47083718891319509963766520854733e-03 1.39465790535856903192835076765732e-04
            8.73165953230075397861753572215093e-01 1.02027227601939069617786870480813e-02 2.69154792147471009274711795100643e-04
            8.18026840363257568000676656083670e-01 1.46381964847189620371192830816653e-02 4.47783878519982176443642307717141e-04
            7.55433500975413618760967437992804e-01 1.96732994769569305426593075480923e-02 6.70515785627624912555722058726815e-04
            6.86853044357709663358946272637695e-01 2.51900152441925165303082678747160e-02 9.26390543861421961006896719226233e-04
            6.13892925570822534098169853677973e-01 3.10590376675150339758069151230302e-02 1.19911732039223008564055028557505e-03
            5.38263260566748669155856532597682e-01 3.71428024304783221953840666174074e-02 1.46841568031313551151439966702128e-03
            4.61736739433251219821841004886664e-01 4.32987116584123490903301956223004e-02 1.71178540594979933936459204346647e-03
            3.86107074429177521412981377579854e-01 4.93824764213756164932256353949924e-02 1.90654274078382230707406908010171e-03
            3.13146955642290225618751264846651e-01 5.52514988446981547554059943649918e-02 2.03193469982910004945075854720926e-03
            2.44566499024586769817091180811985e-01 6.07682146119336991096915312482452e-02 2.07113439255240949776748493604828e-03
            1.81973159636742376488172112658503e-01 6.58033176041716988402541232971998e-02 2.01292999496473744291957785890190e-03
            1.26834046769924602138246427784907e-01 7.02387913286967591197651472612051e-02 1.85294726957825725728590260388273e-03
            8.04415140888906643468203583324794e-02 7.39706768999774666450974791587214e-02 1.59428813137244343445253491609037e-03
            4.38827858743368603988699305773480e-02 7.69115163507204241399861643913027e-02 1.24752356835865826217846219492458e-03
            1.80140363610431508689657675859053e-02 7.89924377291560259051195203028328e-02 8.30061633008494573204849587000353e-04
            3.43570040745255766978516476228833e-03 8.01651411461593577101325536204968e-02 3.65449376941065604169900371545054e-04
            9.96564299592547442330214835237712e-01 1.50768105308612487911595900769157e-04 9.48172493430290239242415935988495e-07
            9.81985963638956738108731769898441e-01 7.90506100364179813654030848368848e-04 1.14595156115234935925847534199384e-05
            9.56117214125662862045373913133517e-01 1.92569889609291080428965514670381e-03 4.30905440763553552996865947388727e-05
            9.19558485911109335653179641667521e-01 3.52999773817024064051506293537841e-03 1.04958558655505462277844619389811e-04
            8.73165953230075397861753572215093e-01 5.56583131598022834429784211351944e-03 2.02559343983050590357747333669636e-04
            8.18026840363257568000676656083670e-01 7.98548919921568574331605105953713e-03 3.36991245652788393681598133611033e-04
            7.55433500975413618760967437992804e-01 1.07322593087321380145304772213422e-02 5.04613856522367031087750177675844e-04
            6.86853044357709663358946272637695e-01 1.37417608016510896934425645099509e-02 6.97178970285089265844502559588136e-04
            6.13892925570822534098169853677973e-01 1.69434540717422407740677670062723e-02 9.02426502754897401706313253555436e-04
            5.38263260566748669155856532597682e-01 2.02622944668638393928006280475529e-02 1.10509389234906998771001962467153e-03
            4.61736739433251219821841004886664e-01 2.36204914074730244755162544834093e-02 1.28824802301484575561829615253373e-03
            3.86107074429177521412981377579854e-01 2.69393318025946161553552116174615e-02 1.43481765183368632603222447841063e-03
            3.13146955642290225618751264846651e-01 3.01410250726857759095977939978184e-02 1.52918458753752714920914890228687e-03
            2.44566499024586769817091180811985e-01 3.31505265656047015676577416343207e-02 1.55868532196257202938893815513666e-03
            1.81973159636742376488172112658503e-01 3.58972966751211763902773554946180e-02 1.51488210932711653451676614423604e-03
            1.26834046769924602138246427784907e-01 3.83169545583566320545720884638285e-02 1.39448300498885694766182385961883e-03
            8.04415140888906643468203583324794e-02 4.03527881361666201920357366361714e-02 1.19982243464506724457685482576608e-03
            4.38827858743368603988699305773480e-02 4.19570869782439626050063452566974e-02 9.38855866521856813680657616316694e-04
            1.80140363610431508689657675859053e-02 4.30922797739726887167321933702624e-02 6.24684177109421554484336436274816e-04
            3.43570040745255766978516476228833e-03 4.37320177690282488325301812892576e-02 2.75028304202134046829719693505467e-04
            9.96564299592547442330214835237712e-01 6.18908320655011404349057912988030e-05 6.14263614385749421766675076178599e-07
            9.81985963638956738108731769898441e-01 3.24505506016986773089316731244480e-04 7.42392710969511182512786326426202e-06
            9.56117214125662862045373913133517e-01 7.90506100364179922074248096919291e-04 2.79157574529834050995440175579887e-05
            9.19558485911109335653179641667521e-01 1.44907635973464126062648649195808e-03 6.79963024103371867570833053662227e-05
            8.73165953230075397861753572215093e-01 2.28479313033166930857476017990848e-03 1.31225948469029991307055471594367e-04
            8.18026840363257568000676656083670e-01 3.27807111443018713708985600874257e-03 2.18316247312922611170660225354823e-04
            7.55433500975413618760967437992804e-01 4.40562980612192300194918104239150e-03 3.26908799321069831224362278732087e-04
            6.86853044357709663358946272637695e-01 5.64104064529018494472678568740776e-03 4.51660090467008445980212627546280e-04
            6.13892925570822534098169853677973e-01 6.95534687802319695482111683304538e-03 5.84627553678838019109165191622424e-04
            5.38263260566748669155856532597682e-01 8.31774241338009701074440727097681e-03 7.15923498364870892346489839042079e-04
            4.61736739433251219821841004886664e-01 9.69629394766305732766831226854265e-03 8.34577982725010826016431852991673e-04
            3.86107074429177521412981377579854e-01 1.10586894830199521794211747760528e-02 9.29531580916538547817995485900155e-04
            3.13146955642290225618751264846651e-01 1.23729957157529685263241958637082e-02 9.90666211382604201471369087528274e-04
            2.44566499024586769817091180811985e-01 1.36084065549212200607609446478818e-02 1.00977795305463185528138669866394e-03
            1.81973159636742376488172112658503e-01 1.47359652466129628645141735887591e-02 9.81400500743376345141877337141523e-04
            1.26834046769924602138246427784907e-01 1.57292432307114815603910074059968e-02 9.03401202607162674695917559120062e-04
            8.04415140888906643468203583324794e-02 1.65649600013085089578179776026445e-02 7.77292391872548473624737752629699e-04
            4.38827858743368603988699305773480e-02 1.72235302606789757173810784252055e-02 6.08227935267961739539133159127005e-04
            1.80140363610431508689657675859053e-02 1.76895308550261663727010130742201e-02 4.04695098349245794243367324583005e-04
            3.43570040745255766978516476228833e-03 1.79521455289776490915620854593726e-02 1.78174204976561811756066711076585e-04
            9.96564299592547442330214835237712e-01 1.18040372897696700763307472992203e-05 2.66484302210673163942164726539219e-07
            9.81985963638956738108731769898441e-01 6.18908320655015199056661612253549e-05 3.22070197413258743848892516614679e-06
            9.56117214125662862045373913133517e-01 1.50768105308613436588496825585537e-04 1.21106166331854691090540165898659e-05
            9.19558485911109335653179641667521e-01 2.76372942731302299879114769964872e-04 2.94986497268671577138589301458893e-05
            8.73165953230075397861753572215093e-01 4.35763786166286728497587521502510e-04 5.69293939779782083012923954434825e-05
            8.18026840363257568000676656083670e-01 6.25205258709385066143160436524795e-04 9.47115399055722305627150192641750e-05
            7.55433500975413618760967437992804e-01 8.40257220348017029240561548419919e-04 1.41821949458488592681060236344592e-04
            6.86853044357709663358946272637695e-01 1.07587912309274483582677284232432e-03 1.95942460575119738663482649165815e-04
            6.13892925570822534098169853677973e-01 1.32654823293664012213988190325153e-03 2.53627371126366972769322627456745e-04
            5.38263260566748669155856532597682e-01 1.58638910380663701246706231984263e-03 3.10587131371544691513242320013433e-04
            4.61736739433251219821841004886664e-01 1.84931130364592109099897143664748e-03 3.62062681491012565166559555862591e-04
            3.86107074429177521412981377579854e-01 2.10915217451591733080484836193591e-03 4.03256141047891902059324698015530e-04
            3.13146955642290225618751264846651e-01 2.35982128435981326763926091416579e-03 4.29778010419803058878018831023837e-04
            2.44566499024586769817091180811985e-01 2.59544318710453912266156173416221e-03 4.38069204988756732543214722497282e-04
            1.81973159636742376488172112658503e-01 2.81049514874317238680156982866265e-03 4.25758292539150549286391944647789e-04
            1.26834046769924602138246427784907e-01 2.99993662128627104970779448933627e-03 3.91920070560894016290648389500006e-04
            8.04415140888906643468203583324794e-02 3.15932746472125536990604999232346e-03 3.37210630437475991985629697467175e-04
            4.38827858743368603988699305773480e-02 3.28493230214394480240280849159262e-03 2.63865860062378963356993777367165e-04
            1.80140363610431508689657675859053e-02 3.37380957538705632606235162995745e-03 1.75567766616815040337309938855981e-04
            3.43570040745255766978516476228833e-03 3.42389637016278806577740390082454e-03 7.72968275723121651336194881665165e-05
            ];
        
    case  38
        
        % ALG. DEG.:   38
        % PTS CARD.:  400
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.5e-15
        
        xyw=[
            9.96564299592547442330214835237712e-01 3.42389637016278806577740390082454e-03 2.66484302210678352018966659128796e-07
            9.81985963638956738108731769898441e-01 1.79521455289777601138645479750267e-02 3.22070197413265011892702198437188e-06
            9.56117214125662862045373913133517e-01 4.37320177690285263882863375783927e-02 1.21106166331857045842133532853602e-05
            9.19558485911109335653179641667521e-01 8.01651411461593577101325536204968e-02 2.94986497268677303081312740529185e-05
            8.73165953230075397861753572215093e-01 1.26398282983758325492829044378595e-01 5.69293939779793196085191930855274e-05
            8.18026840363257568000676656083670e-01 1.81347954378032999445125028614711e-01 9.47115399055740737064082446217128e-05
            7.55433500975413618760967437992804e-01 2.43726241804238352717248972112429e-01 1.41821949458491357396600074380899e-04
            6.86853044357709663358946272637695e-01 3.12071076519197576626396539722919e-01 1.95942460575123560476140660568944e-04
            6.13892925570822534098169853677973e-01 3.84780526196240812986104629089823e-01 2.53627371126371905889207436501920e-04
            5.38263260566748669155856532597682e-01 4.60150350329444679520207728273817e-01 3.10587131371550763045408238838263e-04
            4.61736739433251219821841004886664e-01 5.36413949263102818321158338221721e-01 3.62062681491019612480680711641412e-04
            3.86107074429177521412981377579854e-01 6.11783773396306518321807743632235e-01 4.03256141047899762525075217922677e-04
            3.13146955642290225618751264846651e-01 6.84493223073349921214969526772620e-01 4.29778010419811461444855593683201e-04
            2.44566499024586769817091180811985e-01 7.52838057788308590012604781804839e-01 4.38069204988765297740377357982311e-04
            1.81973159636742376488172112658503e-01 8.15216345214514359618362959736260e-01 4.25758292539158843433011458756710e-04
            1.26834046769924602138246427784907e-01 8.70166016608789116837385790859116e-01 3.91920070560901659915964412306266e-04
            8.04415140888906643468203583324794e-02 9.16399158446388084620082281617215e-01 3.37210630437482551408773234769001e-04
            4.38827858743368603988699305773480e-02 9.52832281823519089414276095340028e-01 2.63865860062384113317313083513227e-04
            1.80140363610431508689657675859053e-02 9.78612154063569827933122269314481e-01 1.75567766616818455574153268194948e-04
            3.43570040745255766978516476228833e-03 9.93140403222384615666840090852929e-01 7.72968275723136830166609678727241e-05
            9.96564299592547442330214835237712e-01 3.37380957538705632606235162995745e-03 6.14263614385752492261108873017328e-07
            9.81985963638956738108731769898441e-01 1.76895308550262739255565236362600e-02 7.42392710969514994161048970777728e-06
            9.56117214125662862045373913133517e-01 4.30922797739729523947005418449407e-02 2.79157574529835474010791562804457e-05
            9.19558485911109335653179641667521e-01 7.89924377291560120273317124883761e-02 6.79963024103375391227893631551638e-05
            8.73165953230075397861753572215093e-01 1.24549253639592918951883859790541e-01 1.31225948469030668933413275034638e-04
            8.18026840363257568000676656083670e-01 1.78695088522312167667038806939672e-01 2.18316247312923722477887022996867e-04
            7.55433500975413618760967437992804e-01 2.40160869218464440022486883208330e-01 3.26908799321071511737729631263960e-04
            6.86853044357709663358946272637695e-01 3.07505914997000140420624347825651e-01 4.51660090467010777014883471380813e-04
            6.13892925570822534098169853677973e-01 3.79151727551154227313645606045611e-01 5.84627553678841054875248151034839e-04
            5.38263260566748669155856532597682e-01 4.53418997019871206077823444502428e-01 7.15923498364874578633876289757154e-04
            4.61736739433251219821841004886664e-01 5.28566966619085643053210787911667e-01 8.34577982725015054404904546458965e-04
            3.86107074429177521412981377579854e-01 6.02834236087802399772783701337175e-01 9.29531580916543318307554422119665e-04
            3.13146955642290225618751264846651e-01 6.74480048641956764221561115846271e-01 9.90666211382609188801362520848670e-04
            2.44566499024586769817091180811985e-01 7.41825094420491826241459420998581e-01 1.00977795305463705945181462908522e-03
            1.81973159636742376488172112658503e-01 8.03290875116644542686117347329855e-01 9.81400500743381332471870770461919e-04
            1.26834046769924602138246427784907e-01 8.57436709999363833034635717922356e-01 9.03401202607167336765259246789128e-04
            8.04415140888906643468203583324794e-02 9.02993525909800753836975673038978e-01 7.77292391872552485172775948996104e-04
            4.38827858743368603988699305773480e-02 9.38893683864983952247484921826981e-01 6.08227935267964883725433367089863e-04
            1.80140363610431508689657675859053e-02 9.64296432783930623777735036128433e-01 4.04695098349247854227495047041430e-04
            3.43570040745255766978516476228833e-03 9.78612154063569716910819806798827e-01 1.78174204976562733327913323755354e-04
            9.96564299592547442330214835237712e-01 3.28493230214394436872193949739085e-03 9.48172493430292145066547258164258e-07
            9.81985963638956738108731769898441e-01 1.72235302606790763313426850800170e-02 1.14595156115235156154413820317473e-05
            9.56117214125662862045373913133517e-01 4.19570869782442124051868859169190e-02 4.30905440763554433911131091861080e-05
            9.19558485911109335653179641667521e-01 7.69115163507203963844105487623892e-02 1.04958558655505679118279116490697e-04
            8.73165953230075397861753572215093e-01 1.21268215453944339099479066135245e-01 2.02559343983050996933562015733798e-04
            8.18026840363257568000676656083670e-01 1.73987670437526636968428306317946e-01 3.36991245652789044202901624913693e-04
            7.55433500975413618760967437992804e-01 2.33834239715854180774456949620799e-01 5.04613856522368006869705414629834e-04
            6.86853044357709663358946272637695e-01 2.99405194840639155007266936081578e-01 6.97178970285090675307326790743900e-04
            6.13892925570822534098169853677973e-01 3.69163620357435107166566012892872e-01 9.02426502754899244850006478912974e-04
            5.38263260566748669155856532597682e-01 4.41474444966387380429040376839112e-01 1.10509389234907215611436459568040e-03
            4.61736739433251219821841004886664e-01 5.14642769159275648149787230067886e-01 1.28824802301484835770351011774437e-03
            3.86107074429177521412981377579854e-01 5.86953593768227643856505437724991e-01 1.43481765183368936179830743782304e-03
            3.13146955642290225618751264846651e-01 6.56712019285023762549258208309766e-01 1.52918458753753018497523186169929e-03
            2.44566499024586769817091180811985e-01 7.22282974409808264937282729079016e-01 1.55868532196257506515502111454907e-03
            1.81973159636742376488172112658503e-01 7.82129543688136141810218759928830e-01 1.51488210932711957028284910364846e-03
            1.26834046769924602138246427784907e-01 8.34848998671718578457046078256099e-01 1.39448300498885976658747232193036e-03
            8.04415140888906643468203583324794e-02 8.79205697774942507294326787814498e-01 1.19982243464506962982163429387583e-03
            4.38827858743368603988699305773480e-02 9.14160127147418899440367567876820e-01 9.38855866521858765244568090224675e-04
            1.80140363610431508689657675859053e-02 9.38893683864983841225182459311327e-01 6.24684177109422855526943418880137e-04
            3.43570040745255766978516476228833e-03 9.52832281823518867369671170308720e-01 2.75028304202134588930805936257684e-04
            9.96564299592547442330214835237712e-01 3.15932746472125536990604999232346e-03 1.25990322327730073706649957521186e-06
            9.81985963638956738108731769898441e-01 1.65649600013086130412265362110702e-02 1.52270612743908618662075013516777e-05
            9.56117214125662862045373913133517e-01 4.03527881361668769311101812036213e-02 5.72574249419144446762372502313099e-05
            9.19558485911109335653179641667521e-01 7.39706768999774666450974791587214e-02 1.39465790535857906079844625857334e-04
            8.73165953230075397861753572215093e-01 1.16631324009730696911191216713632e-01 2.69154792147472960838622269008624e-04
            8.18026840363257568000676656083670e-01 1.67334963152023424859393685437681e-01 4.47783878519985374840051139955222e-04
            7.55433500975413618760967437992804e-01 2.24893199547629457635267158366332e-01 6.70515785627629683045280994946324e-04
            6.86853044357709663358946272637695e-01 2.87956940398097827049639363394817e-01 9.26390543861428574640148880803281e-04
            6.13892925570822534098169853677973e-01 3.55048036761662411109341519477312e-01 1.19911732039223854241749567250963e-03
            5.38263260566748669155856532597682e-01 4.24593937002772980893183785155998e-01 1.46841568031314591985525552786385e-03
            4.61736739433251219821841004886664e-01 4.94964548908336410271147087769350e-01 1.71178540594981148242892388111613e-03
            3.86107074429177521412981377579854e-01 5.64510449149446813521535659674555e-01 1.90654274078383596802144239745758e-03
            3.13146955642290225618751264846651e-01 6.31601545513011619625842740788357e-01 2.03193469982911436091943535586779e-03
            2.44566499024586769817091180811985e-01 6.94665286363479461684278248867486e-01 2.07113439255242424291703073890858e-03
            1.81973159636742376488172112658503e-01 7.52223522759085883038210340600926e-01 2.01292999496475175438825466756043e-03
            1.26834046769924602138246427784907e-01 8.02927161901378694253139656211715e-01 1.85294726957827070139284142413771e-03
            8.04415140888906643468203583324794e-02 8.45587809011131841252506546879886e-01 1.59428813137245492699556326243737e-03
            4.38827858743368603988699305773480e-02 8.79205697774942618316629250330152e-01 1.24752356835866715263627657606094e-03
            1.80140363610431508689657675859053e-02 9.02993525909800864859278135554632e-01 8.30061633008500536316798257274741e-04
            3.43570040745255766978516476228833e-03 9.16399158446388084620082281617215e-01 3.65449376941068206255114336755696e-04
            9.96564299592547442330214835237712e-01 2.99993662128627104970779448933627e-03 1.54211228820107622252604060580916e-06
            9.81985963638956738108731769898441e-01 1.57292432307115787049056621071941e-02 1.86378111196089936939506576107917e-05
            9.56117214125662862045373913133517e-01 3.83169545583568749158587252168218e-02 7.00826674321818803101852290993179e-05
            9.19558485911109335653179641667521e-01 7.02387913286967591197651472612051e-02 1.70705102896373143517641524091744e-04
            8.73165953230075397861753572215093e-01 1.10747171349889178548764334664156e-01 3.29443487984052590412942240760685e-04
            8.18026840363257568000676656083670e-01 1.58892767396504847043914310233959e-01 5.48084177233685199311097324681441e-04
            7.55433500975413618760967437992804e-01 2.13547140248945277463121783512179e-01 8.20706395019344856454901471209951e-04
            6.86853044357709663358946272637695e-01 2.73429260024496578740382801697706e-01 1.13389521906759739404424447428710e-03
            6.13892925570822534098169853677973e-01 3.37135551692828439040994226161274e-01 1.46771078969183720389080072266097e-03
            5.38263260566748669155856532597682e-01 4.03172800228581817982842494529905e-01 1.79733000357571478904084028016541e-03
            4.61736739433251219821841004886664e-01 4.69993153001493635390062308943016e-01 2.09521275960532086038434762542693e-03
            3.86107074429177521412981377579854e-01 5.36030401537246903309608114795992e-01 2.33359430647016048016029188261200e-03
            3.13146955642290225618751264846651e-01 5.99736693205578874632522001775214e-01 2.48707314302899860219975280983817e-03
            2.44566499024586769817091180811985e-01 6.59618812981129787331724401155952e-01 2.53505327890409925400971857811783e-03
            1.81973159636742376488172112658503e-01 7.14273185833570467551112415094394e-01 2.46381152391135461662829975182376e-03
            1.26834046769924602138246427784907e-01 7.62418781880186191557413621922024e-01 2.26799384350520605302392596058780e-03
            8.04415140888906643468203583324794e-02 8.02927161901378694253139656211715e-01 1.95139695882932950876664790484938e-03
            4.38827858743368603988699305773480e-02 8.34848998671718689479348540771753e-01 1.52695968153970578970457516732040e-03
            1.80140363610431508689657675859053e-02 8.57436709999363944056938180438010e-01 1.01598933995657061783723928982681e-03
            3.43570040745255766978516476228833e-03 8.70166016608789116837385790859116e-01 4.47307352250665409085922386012157e-04
            9.96564299592547442330214835237712e-01 2.81049514874317238680156982866265e-03 1.78817841539713588027609228459092e-06
            9.81985963638956738108731769898441e-01 1.47359652466130530701349243827281e-02 2.16117411224388875740868437569375e-05
            9.56117214125662862045373913133517e-01 3.58972966751213984348822805259260e-02 8.12653619029737639455415743050537e-05
            9.19558485911109335653179641667521e-01 6.58033176041716988402541232971998e-02 1.97943549722651471970552572443580e-04
            8.73165953230075397861753572215093e-01 1.03753654529687058816200817545905e-01 3.82010920225165338543621418310181e-04
            8.18026840363257568000676656083670e-01 1.48858928808563051315871916813194e-01 6.35538866429279395917073536281805e-04
            7.55433500975413618760967437992804e-01 2.00061960455786114199128178370302e-01 9.51661868062769365425590528673183e-04
            6.86853044357709663358946272637695e-01 2.56162614693435930490750251919962e-01 1.31482445965329375336949624397676e-03
            6.13892925570822534098169853677973e-01 3.15845950137201181284751783095999e-01 1.70190509099309877395767021113215e-03
            5.38263260566748669155856532597682e-01 3.77713046038215316535513466078555e-01 2.08411977670505633650210874918685e-03
            4.61736739433251219821841004886664e-01 4.40313794325042306976314421262941e-01 2.42953399765814034155786416135925e-03
            3.86107074429177521412981377579854e-01 5.02180890226056275693622410472017e-01 2.70595273836472241177353303953623e-03
            3.13146955642290225618751264846651e-01 5.61864225669821748532228866679361e-01 2.88392132395645693979102652804158e-03
            2.44566499024586769817091180811985e-01 6.17964879907471065223489858908579e-01 2.93955737847474479415565973283719e-03
            1.81973159636742376488172112658503e-01 6.69167911554694461173653508012649e-01 2.85694797997131620659327921885051e-03
            1.26834046769924602138246427784907e-01 7.14273185833570467551112415094394e-01 2.62988478091991714075126296279450e-03
            8.04415140888906643468203583324794e-02 7.52223522759085883038210340600926e-01 2.26277032376207302630244377894542e-03
            4.38827858743368603988699305773480e-02 7.82129543688136363854823684960138e-01 1.77060799307693426778154144329847e-03
            1.80140363610431508689657675859053e-02 8.03290875116644653708419809845509e-01 1.17810500693386194237533004525176e-03
            3.43570040745255766978516476228833e-03 8.15216345214514359618362959736260e-01 5.18681654029326359711671301511160e-04
            9.96564299592547442330214835237712e-01 2.59544318710454085738503771096930e-03 1.99233227561257533345787963663387e-06
            9.81985963638956738108731769898441e-01 1.36084065549213120011051714186578e-02 2.40791237606208626464629474206447e-05
            9.56117214125662862045373913133517e-01 3.31505265656049305511565705728572e-02 9.05433160441505570066936248174727e-05
            9.19558485911109335653179641667521e-01 6.07682146119337338041610507843870e-02 2.20542491434880586192307516668620e-04
            8.73165953230075397861753572215093e-01 9.58146879942834983623356492898893e-02 4.25624579431033362494213312388069e-04
            8.18026840363257568000676656083670e-01 1.37468621067942109448267729021609e-01 7.08097461131736074374520839569414e-04
            7.55433500975413618760967437992804e-01 1.84753726579443361410781676568149e-01 1.06031178929655279206956119963934e-03
            6.86853044357709663358946272637695e-01 2.36561701020647940918451013203594e-01 1.46493615244222877801172355560766e-03
            6.13892925570822534098169853677973e-01 2.91678218987408144347028837728431e-01 1.89620924490460011739567658395345e-03
            5.38263260566748669155856532597682e-01 3.48811401599033399367755237108213e-01 2.32206085344665311490097714397507e-03
            4.61736739433251219821841004886664e-01 4.06622099376380330415514663400245e-01 2.70691053898967264393360387941811e-03
            3.86107074429177521412981377579854e-01 4.63755281988005418902787369006546e-01 3.01488762558905316543689245634141e-03
            3.13146955642290225618751264846651e-01 5.18871799954765733353667656047037e-01 3.21317463882357179719151396568577e-03
            2.44566499024586769817091180811985e-01 5.70679774395969952038853989506606e-01 3.27516258485294371738394403337225e-03
            1.81973159636742376488172112658503e-01 6.17964879907471509312699708971195e-01 3.18312178540577613364281894803298e-03
            1.26834046769924602138246427784907e-01 6.59618812981130120398631788702914e-01 2.93013509449245633867175975240116e-03
            8.04415140888906643468203583324794e-02 6.94665286363479905773488098930102e-01 2.52110768674515756462839455309677e-03
            4.38827858743368603988699305773480e-02 7.22282974409808820048795041657286e-01 1.97275586244079073963697545934792e-03
            1.80140363610431508689657675859053e-02 7.41825094420492381352971733576851e-01 1.31260762861508221976958310506234e-03
            3.43570040745255766978516476228833e-03 7.52838057788309034101814631867455e-01 5.77898822171633158584214129405154e-04
            9.96564299592547442330214835237712e-01 2.35982128435981283395839191996401e-03 2.14978807714870044011270903971944e-06
            9.81985963638956738108731769898441e-01 1.23729957157530431194336628664132e-02 2.59821184460080637705463313391618e-05
            9.56117214125662862045373913133517e-01 3.01410250726859597902862475393704e-02 9.76990352863572924157695154256942e-05
            9.19558485911109335653179641667521e-01 5.52514988446981478165120904577634e-02 2.37972161769853295457122221989721e-04
            8.73165953230075397861753572215093e-01 8.71163511521308442375755021203076e-02 4.59262070590576076869332444729821e-04
            8.18026840363257568000676656083670e-01 1.24988818687887998093444252845075e-01 7.64059036754914742278765871219548e-04
            7.55433500975413618760967437992804e-01 1.67981244402943985516429847848485e-01 1.14410917826901210696266542043986e-03
            6.86853044357709663358946272637695e-01 2.15085939814255777680074288582546e-01 1.58071136670016052297449427044285e-03
            6.13892925570822534098169853677973e-01 2.65198819519729345195457881345646e-01 2.04606835735855318442188988115049e-03
            5.38263260566748669155856532597682e-01 3.17145285171531232659702936871327e-01 2.50557539937383035033580114259166e-03
            4.61736739433251219821841004886664e-01 3.69707759186178541721545798282023e-01 2.92084009974637702489186352750039e-03
            3.86107074429177521412981377579854e-01 4.21654224837980262652337160034222e-01 3.25315689093160897299439326957327e-03
            3.13146955642290225618751264846651e-01 4.71767104543453941190023215312976e-01 3.46711470415527345939610270875164e-03
            2.44566499024586769817091180811985e-01 5.18871799954765400286760268500075e-01 3.53400161299677589149537126900213e-03
            1.81973159636742376488172112658503e-01 5.61864225669821637509926404163707e-01 3.43468674685482250677481808054381e-03
            1.26834046769924602138246427784907e-01 5.99736693205578763610219539259560e-01 3.16170629150605954696406918458251e-03
            8.04415140888906643468203583324794e-02 6.31601545513011508603540278272703e-01 2.72035308192066470345249484807937e-03
            4.38827858743368603988699305773480e-02 6.56712019285023873571560670825420e-01 2.12866452253625531437775109111499e-03
            1.80140363610431508689657675859053e-02 6.74480048641956653199258653330617e-01 1.41634418340360213744988371331601e-03
            3.43570040745255766978516476228833e-03 6.84493223073349810192667064256966e-01 6.23570682917772944713064031674321e-04
            9.96564299592547442330214835237712e-01 2.10915217451591776448571735613768e-03 2.25685495042602009014426345245852e-06
            9.81985963638956738108731769898441e-01 1.10586894830200215683602138483366e-02 2.72761177070071468434356298793375e-05
            9.56117214125662862045373913133517e-01 2.69393318025947896277028092981709e-02 1.02564784771857513329985922556631e-04
            9.19558485911109335653179641667521e-01 4.93824764213756303710134432094492e-02 2.49823997566401103439387432203489e-04
            8.73165953230075397861753572215093e-01 7.78625240335755475218348919952405e-02 4.82134908353365238160848083026622e-04
            8.18026840363257568000676656083670e-01 1.11712035344766091871093749432475e-01 8.02111816437798327145136667581937e-04
            7.55433500975413618760967437992804e-01 1.50137643582817059684231253413600e-01 1.20108976798723936056656480531046e-03
            6.86853044357709663358946272637695e-01 1.92238700732842215934681462385925e-01 1.65943625376483014019490802581913e-03
            6.13892925570822534098169853677973e-01 2.37028401504919078446320668263070e-01 2.14796963026197022458840812930703e-03
            5.38263260566748669155856532597682e-01 2.83456917814211228900944661290850e-01 2.63036171046330787340328249968024e-03
            4.61736739433251219821841004886664e-01 3.30436007756611360708376423644950e-01 3.06630802756074644782602334203148e-03
            3.86107074429177521412981377579854e-01 3.76864524065903372385122338528163e-01 3.41517534302693227463154634904186e-03
            3.13146955642290225618751264846651e-01 4.21654224837980373674639622549876e-01 3.63978899452536020647785974801991e-03
            2.44566499024586769817091180811985e-01 4.63755281988005196858182443975238e-01 3.71000710250642736637294305523938e-03
            1.81973159636742376488172112658503e-01 5.02180890226056386715924872987671e-01 3.60574601292003218272474995842458e-03
            1.26834046769924602138246427784907e-01 5.36030401537247014331910577311646e-01 3.31917018780869413699385894744864e-03
            8.04415140888906643468203583324794e-02 5.64510449149446924543838122190209e-01 2.85583606360966402865075863815036e-03
            4.38827858743368603988699305773480e-02 5.86953593768227865901110362756299e-01 2.23467936981673522683911059516504e-03
            1.80140363610431508689657675859053e-02 6.02834236087802621817388626368484e-01 1.48688301688837507148754912833510e-03
            3.43570040745255766978516476228833e-03 6.11783773396306629344110206147889e-01 6.54626657223835090905483813372712e-04
            9.96564299592547442330214835237712e-01 1.84931130364592065731810244244571e-03 2.31102323612482596473837251349615e-06
            9.81985963638956738108731769898441e-01 9.69629394766311457354301950317677e-03 2.79307900582047799887663169782215e-05
            9.56117214125662862045373913133517e-01 2.36204914074731701922882365352052e-02 1.05026510795991137050230113114679e-04
            9.19558485911109335653179641667521e-01 4.32987116584123421514362917150720e-02 2.55820190486127820982814595751620e-04
            8.73165953230075397861753572215093e-01 6.82701075652551170325210705414065e-02 4.93706950879236358513402205261400e-04
            8.18026840363257568000676656083670e-01 9.79494662417064176906933425925672e-02 8.21363838827180977111841464477493e-04
            7.55433500975413618760967437992804e-01 1.31641161190368422007068716084177e-01 1.22991792714295471687646621461454e-03
            6.86853044357709663358946272637695e-01 1.68555501380570210701037581202399e-01 1.69926549359963722542199437981481e-03
            6.13892925570822534098169853677973e-01 2.07827252810137363958631340210559e-01 2.19952448653770249800687430763446e-03
            5.38263260566748669155856532597682e-01 2.48535922890801092011159312278323e-01 2.69349478181850560468135746816642e-03
            4.61736739433251219821841004886664e-01 2.89727337675947604900272835948272e-01 3.13990453815886838215276632979567e-03
            3.86107074429177521412981377579854e-01 3.30436007756611249686073961129296e-01 3.49714525148634792009461058626130e-03
            3.13146955642290225618751264846651e-01 3.69707759186178541721545798282023e-01 3.72715000552059552013650645108100e-03
            2.44566499024586769817091180811985e-01 4.06622099376379997348607275853283e-01 3.79905346529338079977766717831855e-03
            1.81973159636742376488172112658503e-01 4.40313794325042251465163190005114e-01 3.69228993553602155563986286779254e-03
            1.26834046769924602138246427784907e-01 4.69993153001493579878911077685189e-01 3.39883581230185968097279314292791e-03
            8.04415140888906643468203583324794e-02 4.94964548908336354759995856511523e-01 2.92438089577681760086735884840436e-03
            4.38827858743368603988699305773480e-02 5.14642769159275648149787230067886e-01 2.28831540456793207372054865800237e-03
            1.80140363610431508689657675859053e-02 5.28566966619085643053210787911667e-01 1.52257069103168152929272949336337e-03
            3.43570040745255766978516476228833e-03 5.36413949263102707298855875706067e-01 6.70338789626434257765130730888359e-04
            9.96564299592547442330214835237712e-01 1.58638910380663657878619332564085e-03 2.31102323612481410627711095329140e-06
            9.81985963638956738108731769898441e-01 8.31774241338014558300173462157545e-03 2.79307900582046342990993892385632e-05
            9.56117214125662862045373913133517e-01 2.02622944668639642928908983776637e-02 1.05026510795990581396616714293657e-04
            9.19558485911109335653179641667521e-01 3.71428024304783152564901627101790e-02 2.55820190486126465730098988871077e-04
            8.73165953230075397861753572215093e-01 5.85639392046694712279375494290434e-02 4.93706950879233756428188240050758e-04
            8.18026840363257568000676656083670e-01 8.40236933950359449196909622514795e-02 8.21363838827176640303151522459757e-04
            7.55433500975413618760967437992804e-01 1.12925337834217931476388230294106e-01 1.22991792714294821166343130158793e-03
            6.86853044357709663358946272637695e-01 1.44591454261720098184440530530992e-01 1.69926549359962811812374550157756e-03
            6.13892925570822534098169853677973e-01 1.78279821619040046432047574853641e-01 2.19952448653769078862341146418657e-03
            5.38263260566748669155856532597682e-01 2.13200816542450183321832923866168e-01 2.69349478181849129321268065950790e-03
            4.61736739433251219821841004886664e-01 2.48535922890801092011159312278323e-01 3.13990453815885190227974455012827e-03
            3.86107074429177521412981377579854e-01 2.83456917814211117878642198775196e-01 3.49714525148632927181724383558503e-03
            3.13146955642290225618751264846651e-01 3.17145285171531177148551705613500e-01 3.72715000552057600449740171200119e-03
            2.44566499024586769817091180811985e-01 3.48811401599033066300847849561251e-01 3.79905346529336085045769344503697e-03
            1.81973159636742376488172112658503e-01 3.77713046038215261024362234820728e-01 3.69228993553600247368162712291451e-03
            1.26834046769924602138246427784907e-01 4.03172800228581762471691263272078e-01 3.39883581230184146637629538645342e-03
            8.04415140888906643468203583324794e-02 4.24593937002772925382032553898171e-01 2.92438089577680242203694405134229e-03
            4.38827858743368603988699305773480e-02 4.41474444966387380429040376839112e-01 2.28831540456791993065621682035271e-03
            1.80140363610431508689657675859053e-02 4.53418997019871150566672213244601e-01 1.52257069103167350619665310063056e-03
            3.43570040745255766978516476228833e-03 4.60150350329444568497905265758163e-01 6.70338789626430679897961528723727e-04
            9.96564299592547442330214835237712e-01 1.32654823293664012213988190325153e-03 2.25685495042600823168300189225377e-06
            9.81985963638956738108731769898441e-01 6.95534687802324119026975424162629e-03 2.72761177070069977656369131224778e-05
            9.56117214125662862045373913133517e-01 1.69434540717423483269232775683122e-02 1.02564784771856957676372523735608e-04
            9.19558485911109335653179641667521e-01 3.10590376675150409147008190302586e-02 2.49823997566399748186671825322946e-04
            8.73165953230075397861753572215093e-01 4.89715227363490615553054396968946e-02 4.82134908353362636075634117815980e-04
            8.18026840363257568000676656083670e-01 7.02611242919762846170783632260282e-02 8.02111816437793990336446725564201e-04
            7.55433500975413618760967437992804e-01 9.44288554417693493103769242225098e-02 1.20108976798723285535352989228386e-03
            6.86853044357709663358946272637695e-01 1.20908254909448148461947880605294e-01 1.65943625376482124973709364468277e-03
            6.13892925570822534098169853677973e-01 1.49078672924258415211085093687871e-01 2.14796963026195851520494528585914e-03
            5.38263260566748669155856532597682e-01 1.78279821619040129698774421740382e-01 2.63036171046329399561547468522349e-03
            4.61736739433251219821841004886664e-01 2.07827252810137447225358187097299e-01 3.06630802756072996795300156236408e-03
            3.86107074429177521412981377579854e-01 2.37028401504919078446320668263070e-01 3.41517534302691362635417959836559e-03
            3.13146955642290225618751264846651e-01 2.65198819519729400706609112603473e-01 3.63978899452534025715788601473832e-03
            2.44566499024586769817091180811985e-01 2.91678218987407977813575143954949e-01 3.71000710250640741705296932195779e-03
            1.81973159636742376488172112658503e-01 3.15845950137201181284751783095999e-01 3.60574601292001266708564521934477e-03
            1.26834046769924602138246427784907e-01 3.37135551692828439040994226161274e-01 3.31917018780867635607823018517593e-03
            8.04415140888906643468203583324794e-02 3.55048036761662466620492750735139e-01 2.85583606360964841613947484688651e-03
            4.38827858743368603988699305773480e-02 3.69163620357435273700019706666353e-01 2.23467936981672308377477875751538e-03
            1.80140363610431508689657675859053e-02 3.79151727551154338335948068561265e-01 1.48688301688836704839147273560229e-03
            3.43570040745255766978516476228833e-03 3.84780526196240868497255860347650e-01 6.54626657223831513038314611208079e-04
            9.96564299592547442330214835237712e-01 1.07587912309274461898633834522343e-03 2.14978807714870594582686619267164e-06
            9.81985963638956738108731769898441e-01 5.64104064529021790447282924674255e-03 2.59821184460081315331821116831890e-05
            9.56117214125662862045373913133517e-01 1.37417608016511712254459354198843e-02 9.76990352863575499137854807329973e-05
            9.19558485911109335653179641667521e-01 2.51900152441925095914143639674876e-02 2.37972161769853891768317089017160e-04
            8.73165953230075397861753572215093e-01 3.97176956177937440228831178501423e-02 4.59262070590577269491722178784698e-04
            8.18026840363257568000676656083670e-01 5.69843409488543645169400519989722e-02 7.64059036754916693842676345127529e-04
            7.55433500975413618760967437992804e-01 7.65852546216423679670270985297975e-02 1.14410917826901514272874837985228e-03
            6.86853044357709663358946272637695e-01 9.80610158280345173276160153363890e-02 1.58071136670016442610231521825881e-03
            6.13892925570822534098169853677973e-01 1.20908254909448092950796649347467e-01 2.04606835735855838859231781157177e-03
            5.38263260566748669155856532597682e-01 1.44591454261720070428864914902078e-01 2.50557539937383685554883605561827e-03
            4.61736739433251219821841004886664e-01 1.68555501380570182945461965573486e-01 2.92084009974638439746663642893054e-03
            3.86107074429177521412981377579854e-01 1.92238700732842104912378999870270e-01 3.25315689093161721293090415940696e-03
            3.13146955642290225618751264846651e-01 2.15085939814255749924498672953632e-01 3.46711470415528256669435158698889e-03
            2.44566499024586769817091180811985e-01 2.36561701020647718873846088172286e-01 3.53400161299678499879362014723938e-03
            1.81973159636742376488172112658503e-01 2.56162614693435819468447789404308e-01 3.43468674685483118039219796457928e-03
            1.26834046769924602138246427784907e-01 2.73429260024496467718080339182052e-01 3.16170629150606735321971108021444e-03
            8.04415140888906643468203583324794e-02 2.87956940398097716027336900879163e-01 2.72035308192067164234639875530775e-03
            4.38827858743368603988699305773480e-02 2.99405194840639099496115704823751e-01 2.12866452253626095222904801573804e-03
            1.80140363610431508689657675859053e-02 3.07505914997000029398321885309997e-01 1.41634418340360560689683566693020e-03
            3.43570040745255766978516476228833e-03 3.12071076519197465604094077207264e-01 6.23570682917774462596105511380529e-04
            9.96564299592547442330214835237712e-01 8.40257220348018330283168531025240e-04 1.99233227561257448642493238233353e-06
            9.81985963638956738108731769898441e-01 4.40562980612195769641870057853339e-03 2.40791237606208524820675803690406e-05
            9.56117214125662862045373913133517e-01 1.07322593087322230159808000848898e-02 9.05433160441505163491121566110564e-05
            9.19558485911109335653179641667521e-01 1.96732994769569617676818751306200e-02 2.20542491434880477772090268118177e-04
            8.73165953230075397861753572215093e-01 3.10193587756411592870620097528445e-02 4.25624579431033199863887439562404e-04
            8.18026840363257568000676656083670e-01 4.45045385688003294899495188019500e-02 7.08097461131735749113869093918083e-04
            7.55433500975413618760967437992804e-01 5.98127724451431030949777323257877e-02 1.06031178929655235838869220543756e-03
            6.86853044357709663358946272637695e-01 7.65852546216425206226929844888218e-02 1.46493615244222812749042006430500e-03
            6.13892925570822534098169853677973e-01 9.44288554417694880882550023670774e-02 1.89620924490459925003393859554990e-03
            5.38263260566748669155856532597682e-01 1.12925337834218139643205347510957e-01 2.32206085344665224753923915557152e-03
            4.61736739433251219821841004886664e-01 1.31641161190368671807249256744399e-01 2.70691053898967134289099689681279e-03
            3.86107074429177521412981377579854e-01 1.50137643582817253973260562815994e-01 3.01488762558905186439428547373609e-03
            3.13146955642290225618751264846651e-01 1.67981244402944290827761619766534e-01 3.21317463882357049614890698308045e-03
            2.44566499024586769817091180811985e-01 1.84753726579443555699810985970544e-01 3.27516258485294241634133705076692e-03
            1.81973159636742376488172112658503e-01 2.00061960455786419510459950288350e-01 3.18312178540577483260021196542766e-03
            1.26834046769924602138246427784907e-01 2.13547140248945610530029171059141e-01 2.93013509449245503762915276979584e-03
            8.04415140888906643468203583324794e-02 2.24893199547629818457750161542208e-01 2.52110768674515626358578757049145e-03
            4.38827858743368603988699305773480e-02 2.33834239715854597108091184054501e-01 1.97275586244078987227523747094438e-03
            1.80140363610431508689657675859053e-02 2.40160869218464828600545502013119e-01 1.31260762861508156924827961375968e-03
            3.43570040745255766978516476228833e-03 2.43726241804238741295307590917218e-01 5.77898822171632941743779632304268e-04
            9.96564299592547442330214835237712e-01 6.25205258709385066143160436524795e-04 1.78817841539713312741901370811481e-06
            9.81985963638956738108731769898441e-01 3.27807111443020708640982974202416e-03 2.16117411224388536927689535849240e-05
            9.56117214125662862045373913133517e-01 7.98548919921573605029685438694287e-03 8.12653619029736419727971696858049e-05
            9.19558485911109335653179641667521e-01 1.46381964847189620371192830816653e-02 1.97943549722651173814955138929861e-04
            8.73165953230075397861753572215093e-01 2.30803922402375398525986582853875e-02 3.82010920225164796442535175557964e-04
            8.18026840363257568000676656083670e-01 3.31142308281793251723001958453096e-02 6.35538866429278420135118299327814e-04
            7.55433500975413618760967437992804e-01 4.45045385688002531621165758224379e-02 9.51661868062767955962766297517419e-04
            6.86853044357709663358946272637695e-01 5.69843409488543853336217637206573e-02 1.31482445965329180180558577006877e-03
            6.13892925570822534098169853677973e-01 7.02611242919762846170783632260282e-02 1.70190509099309617187245624592151e-03
            5.38263260566748669155856532597682e-01 8.40236933950359587974787700659363e-02 2.08411977670505286705515679557266e-03
            4.61736739433251219821841004886664e-01 9.79494662417064315684811504070240e-02 2.42953399765813643843004321354329e-03
            3.86107074429177521412981377579854e-01 1.11712035344766077993305941618019e-01 2.70595273836471807496484309751850e-03
            3.13146955642290225618751264846651e-01 1.24988818687888011971232060659531e-01 2.88392132395645260298233658602385e-03
            2.44566499024586769817091180811985e-01 1.37468621067942026181540882134868e-01 2.93955737847474045734696979081946e-03
            1.81973159636742376488172112658503e-01 1.48858928808563051315871916813194e-01 2.85694797997131186978458927683278e-03
            1.26834046769924602138246427784907e-01 1.58892767396504847043914310233959e-01 2.62988478091991280394257302077676e-03
            8.04415140888906643468203583324794e-02 1.67334963152023424859393685437681e-01 2.26277032376206955685549182533123e-03
            4.38827858743368603988699305773480e-02 1.73987670437526692479579537575773e-01 1.77060799307693144885589298098694e-03
            1.80140363610431508689657675859053e-02 1.78695088522312195422614422568586e-01 1.17810500693386020765185406844466e-03
            3.43570040745255766978516476228833e-03 1.81347954378032999445125028614711e-01 5.18681654029325492349933313107613e-04
            9.96564299592547442330214835237712e-01 4.35763786166286728497587521502510e-04 1.54211228820109888065737965834323e-06
            9.81985963638956738108731769898441e-01 2.28479313033168362004343698856701e-03 1.86378111196092647444937789869002e-05
            9.56117214125662862045373913133517e-01 5.56583131598026303876736164966132e-03 7.00826674321829103022490903285302e-05
            9.19558485911109335653179641667521e-01 1.02027227601939069617786870480813e-02 1.70705102896375664287692552889553e-04
            8.73165953230075397861753572215093e-01 1.60868754200354201200351411671363e-02 3.29443487984057415112609801255417e-04
            8.18026840363257568000676656083670e-01 2.30803922402375398525986582853875e-02 5.48084177233693222407173717414253e-04
            7.55433500975413618760967437992804e-01 3.10193587756411072453577304486316e-02 8.20706395019356891099016060309168e-04
            6.86853044357709663358946272637695e-01 3.97176956177937579006709256645991e-02 1.13389521906761409075770075105538e-03
            6.13892925570822534098169853677973e-01 4.89715227363490546164115357896662e-02 1.46771078969185867109381593564876e-03
            5.38263260566748669155856532597682e-01 5.85639392046694920446192611507286e-02 1.79733000357574124357384892647360e-03
            4.61736739433251219821841004886664e-01 6.82701075652551309103088783558633e-02 2.09521275960535165172604621375285e-03
            3.86107074429177521412981377579854e-01 7.78625240335755336440470841807837e-02 2.33359430647019474094894242455211e-03
            3.13146955642290225618751264846651e-01 8.71163511521308581153633099347644e-02 2.48707314302903546507361731698893e-03
            2.44566499024586769817091180811985e-01 9.58146879942834428511844180320622e-02 2.53505327890413655056445207947036e-03
            1.81973159636742376488172112658503e-01 1.03753654529687058816200817545905e-01 2.46381152391139104582129526477274e-03
            1.26834046769924602138246427784907e-01 1.10747171349889178548764334664156e-01 2.26799384350523944645083851412437e-03
            8.04415140888906643468203583324794e-02 1.16631324009730696911191216713632e-01 1.95139695882935813170400152216644e-03
            4.38827858743368603988699305773480e-02 1.21268215453944366855054681764159e-01 1.52695968153972812426932836871174e-03
            1.80140363610431508689657675859053e-02 1.24549253639592932829671667604998e-01 1.01598933995658557982721958978800e-03
            3.43570040745255766978516476228833e-03 1.26398282983758325492829044378595e-01 4.47307352250671968509065923313983e-04
            9.96564299592547442330214835237712e-01 2.76372942731302299879114769964872e-04 1.25990322327729163146231659148322e-06
            9.81985963638956738108731769898441e-01 1.44907635973465036792473537019532e-03 1.52270612743907534459902528012343e-05
            9.56117214125662862045373913133517e-01 3.52999773817026319192025063387064e-03 5.72574249419140313241589901327444e-05
            9.19558485911109335653179641667521e-01 6.47083718891319509963766520854733e-03 1.39465790535856903192835076765732e-04
            8.73165953230075397861753572215093e-01 1.02027227601939069617786870480813e-02 2.69154792147471009274711795100643e-04
            8.18026840363257568000676656083670e-01 1.46381964847189620371192830816653e-02 4.47783878519982176443642307717141e-04
            7.55433500975413618760967437992804e-01 1.96732994769569305426593075480923e-02 6.70515785627624912555722058726815e-04
            6.86853044357709663358946272637695e-01 2.51900152441925165303082678747160e-02 9.26390543861421961006896719226233e-04
            6.13892925570822534098169853677973e-01 3.10590376675150339758069151230302e-02 1.19911732039223008564055028557505e-03
            5.38263260566748669155856532597682e-01 3.71428024304783221953840666174074e-02 1.46841568031313551151439966702128e-03
            4.61736739433251219821841004886664e-01 4.32987116584123490903301956223004e-02 1.71178540594979933936459204346647e-03
            3.86107074429177521412981377579854e-01 4.93824764213756164932256353949924e-02 1.90654274078382230707406908010171e-03
            3.13146955642290225618751264846651e-01 5.52514988446981547554059943649918e-02 2.03193469982910004945075854720926e-03
            2.44566499024586769817091180811985e-01 6.07682146119336991096915312482452e-02 2.07113439255240949776748493604828e-03
            1.81973159636742376488172112658503e-01 6.58033176041716988402541232971998e-02 2.01292999496473744291957785890190e-03
            1.26834046769924602138246427784907e-01 7.02387913286967591197651472612051e-02 1.85294726957825725728590260388273e-03
            8.04415140888906643468203583324794e-02 7.39706768999774666450974791587214e-02 1.59428813137244343445253491609037e-03
            4.38827858743368603988699305773480e-02 7.69115163507204241399861643913027e-02 1.24752356835865826217846219492458e-03
            1.80140363610431508689657675859053e-02 7.89924377291560259051195203028328e-02 8.30061633008494573204849587000353e-04
            3.43570040745255766978516476228833e-03 8.01651411461593577101325536204968e-02 3.65449376941065604169900371545054e-04
            9.96564299592547442330214835237712e-01 1.50768105308612487911595900769157e-04 9.48172493430290239242415935988495e-07
            9.81985963638956738108731769898441e-01 7.90506100364179813654030848368848e-04 1.14595156115234935925847534199384e-05
            9.56117214125662862045373913133517e-01 1.92569889609291080428965514670381e-03 4.30905440763553552996865947388727e-05
            9.19558485911109335653179641667521e-01 3.52999773817024064051506293537841e-03 1.04958558655505462277844619389811e-04
            8.73165953230075397861753572215093e-01 5.56583131598022834429784211351944e-03 2.02559343983050590357747333669636e-04
            8.18026840363257568000676656083670e-01 7.98548919921568574331605105953713e-03 3.36991245652788393681598133611033e-04
            7.55433500975413618760967437992804e-01 1.07322593087321380145304772213422e-02 5.04613856522367031087750177675844e-04
            6.86853044357709663358946272637695e-01 1.37417608016510896934425645099509e-02 6.97178970285089265844502559588136e-04
            6.13892925570822534098169853677973e-01 1.69434540717422407740677670062723e-02 9.02426502754897401706313253555436e-04
            5.38263260566748669155856532597682e-01 2.02622944668638393928006280475529e-02 1.10509389234906998771001962467153e-03
            4.61736739433251219821841004886664e-01 2.36204914074730244755162544834093e-02 1.28824802301484575561829615253373e-03
            3.86107074429177521412981377579854e-01 2.69393318025946161553552116174615e-02 1.43481765183368632603222447841063e-03
            3.13146955642290225618751264846651e-01 3.01410250726857759095977939978184e-02 1.52918458753752714920914890228687e-03
            2.44566499024586769817091180811985e-01 3.31505265656047015676577416343207e-02 1.55868532196257202938893815513666e-03
            1.81973159636742376488172112658503e-01 3.58972966751211763902773554946180e-02 1.51488210932711653451676614423604e-03
            1.26834046769924602138246427784907e-01 3.83169545583566320545720884638285e-02 1.39448300498885694766182385961883e-03
            8.04415140888906643468203583324794e-02 4.03527881361666201920357366361714e-02 1.19982243464506724457685482576608e-03
            4.38827858743368603988699305773480e-02 4.19570869782439626050063452566974e-02 9.38855866521856813680657616316694e-04
            1.80140363610431508689657675859053e-02 4.30922797739726887167321933702624e-02 6.24684177109421554484336436274816e-04
            3.43570040745255766978516476228833e-03 4.37320177690282488325301812892576e-02 2.75028304202134046829719693505467e-04
            9.96564299592547442330214835237712e-01 6.18908320655011404349057912988030e-05 6.14263614385749421766675076178599e-07
            9.81985963638956738108731769898441e-01 3.24505506016986773089316731244480e-04 7.42392710969511182512786326426202e-06
            9.56117214125662862045373913133517e-01 7.90506100364179922074248096919291e-04 2.79157574529834050995440175579887e-05
            9.19558485911109335653179641667521e-01 1.44907635973464126062648649195808e-03 6.79963024103371867570833053662227e-05
            8.73165953230075397861753572215093e-01 2.28479313033166930857476017990848e-03 1.31225948469029991307055471594367e-04
            8.18026840363257568000676656083670e-01 3.27807111443018713708985600874257e-03 2.18316247312922611170660225354823e-04
            7.55433500975413618760967437992804e-01 4.40562980612192300194918104239150e-03 3.26908799321069831224362278732087e-04
            6.86853044357709663358946272637695e-01 5.64104064529018494472678568740776e-03 4.51660090467008445980212627546280e-04
            6.13892925570822534098169853677973e-01 6.95534687802319695482111683304538e-03 5.84627553678838019109165191622424e-04
            5.38263260566748669155856532597682e-01 8.31774241338009701074440727097681e-03 7.15923498364870892346489839042079e-04
            4.61736739433251219821841004886664e-01 9.69629394766305732766831226854265e-03 8.34577982725010826016431852991673e-04
            3.86107074429177521412981377579854e-01 1.10586894830199521794211747760528e-02 9.29531580916538547817995485900155e-04
            3.13146955642290225618751264846651e-01 1.23729957157529685263241958637082e-02 9.90666211382604201471369087528274e-04
            2.44566499024586769817091180811985e-01 1.36084065549212200607609446478818e-02 1.00977795305463185528138669866394e-03
            1.81973159636742376488172112658503e-01 1.47359652466129628645141735887591e-02 9.81400500743376345141877337141523e-04
            1.26834046769924602138246427784907e-01 1.57292432307114815603910074059968e-02 9.03401202607162674695917559120062e-04
            8.04415140888906643468203583324794e-02 1.65649600013085089578179776026445e-02 7.77292391872548473624737752629699e-04
            4.38827858743368603988699305773480e-02 1.72235302606789757173810784252055e-02 6.08227935267961739539133159127005e-04
            1.80140363610431508689657675859053e-02 1.76895308550261663727010130742201e-02 4.04695098349245794243367324583005e-04
            3.43570040745255766978516476228833e-03 1.79521455289776490915620854593726e-02 1.78174204976561811756066711076585e-04
            9.96564299592547442330214835237712e-01 1.18040372897696700763307472992203e-05 2.66484302210673163942164726539219e-07
            9.81985963638956738108731769898441e-01 6.18908320655015199056661612253549e-05 3.22070197413258743848892516614679e-06
            9.56117214125662862045373913133517e-01 1.50768105308613436588496825585537e-04 1.21106166331854691090540165898659e-05
            9.19558485911109335653179641667521e-01 2.76372942731302299879114769964872e-04 2.94986497268671577138589301458893e-05
            8.73165953230075397861753572215093e-01 4.35763786166286728497587521502510e-04 5.69293939779782083012923954434825e-05
            8.18026840363257568000676656083670e-01 6.25205258709385066143160436524795e-04 9.47115399055722305627150192641750e-05
            7.55433500975413618760967437992804e-01 8.40257220348017029240561548419919e-04 1.41821949458488592681060236344592e-04
            6.86853044357709663358946272637695e-01 1.07587912309274483582677284232432e-03 1.95942460575119738663482649165815e-04
            6.13892925570822534098169853677973e-01 1.32654823293664012213988190325153e-03 2.53627371126366972769322627456745e-04
            5.38263260566748669155856532597682e-01 1.58638910380663701246706231984263e-03 3.10587131371544691513242320013433e-04
            4.61736739433251219821841004886664e-01 1.84931130364592109099897143664748e-03 3.62062681491012565166559555862591e-04
            3.86107074429177521412981377579854e-01 2.10915217451591733080484836193591e-03 4.03256141047891902059324698015530e-04
            3.13146955642290225618751264846651e-01 2.35982128435981326763926091416579e-03 4.29778010419803058878018831023837e-04
            2.44566499024586769817091180811985e-01 2.59544318710453912266156173416221e-03 4.38069204988756732543214722497282e-04
            1.81973159636742376488172112658503e-01 2.81049514874317238680156982866265e-03 4.25758292539150549286391944647789e-04
            1.26834046769924602138246427784907e-01 2.99993662128627104970779448933627e-03 3.91920070560894016290648389500006e-04
            8.04415140888906643468203583324794e-02 3.15932746472125536990604999232346e-03 3.37210630437475991985629697467175e-04
            4.38827858743368603988699305773480e-02 3.28493230214394480240280849159262e-03 2.63865860062378963356993777367165e-04
            1.80140363610431508689657675859053e-02 3.37380957538705632606235162995745e-03 1.75567766616815040337309938855981e-04
            3.43570040745255766978516476228833e-03 3.42389637016278806577740390082454e-03 7.72968275723121651336194881665165e-05
            ];
        
    case  39
        
        % ALG. DEG.:   39
        % PTS CARD.:  441
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.4e-15
        
        xyw=[
            9.96876085310194670618955115060089e-01 3.11415584681614798084892115070943e-03 2.00361328812385754687079423598528e-07
            9.83613419283153045391543400910450e-01 1.63353904366299144956808930828629e-02 2.42479316225436106088360690979933e-06
            9.60049667075200341415097682329360e-01 3.98255314929132075985052097166772e-02 9.13998831301501451541708515025064e-06
            9.26681682291658592731664612074383e-01 7.30892775386205589827071094077837e-02 2.23421606398037836437378556020406e-05
            8.84219981737838889301883682492189e-01 1.15418331362325959910108963413222e-01 4.33225863344741942773455378734582e-05
            8.33569402098706113690695929108188e-01 1.65910682911676959916391638216737e-01 7.25067870770942143193726892036466e-05
            7.75809417943609802570392730558524e-01 2.23490229803788242879392100803670e-01 1.09371465126862420593790992118244e-04
            7.12171060103719444356329404399730e-01 2.86929786842787526435216705067432e-01 1.52447287063446173778838432788518e-04
            6.44010658401200752898319024097873e-01 3.54877261265164611891975710022962e-01 1.99408171289469762560084276081795e-04
            5.72780927080447255228534686466446e-01 4.25884476981894299907338563571102e-01 2.47240261133546722956827457196027e-04
            5.00000000000000222044604925031308e-01 4.98438042655097113264872632498736e-01 2.92476857727690965887273888412778e-04
            4.27219072919552467215709157244419e-01 5.70991608328300537245070245262468e-01 3.31479830747985731564247968705672e-04
            3.55989341598799580168588363449089e-01 6.41998824045029725660072017490165e-01 3.60743911899029444205477812346317e-04
            2.87828939896280500132519364342443e-01 7.09946298467407199694889641250484e-01 3.77198158312480992873649832120009e-04
            2.24190582056390030896153575667995e-01 7.73385855506406594273016708029900e-01 3.78478934848246791750048245717153e-04
            1.66430597901293941820455302149639e-01 8.30965402398517682946987861214438e-01 3.63151006570300543922769076488066e-04
            1.15780018262161166209267548765638e-01 8.81457753947868627442119304760126e-01 3.30857578643379516159656050788840e-04
            7.33183177083414627794866191834444e-02 9.23786807771574181025187044724589e-01 2.82386061967272380957005850632413e-04
            3.99503329247995475625998551549856e-02 9.57050553817281546287176752230152e-01 2.19643795046674109635448379762579e-04
            1.63865807168467880750029053160688e-02 9.80540694873564877553917540353723e-01 1.45549528275133217706041599548428e-04
            3.12391468980532938104488493991084e-03 9.93761929463378490545721888338448e-01 6.39375389365999047611155425840934e-05
            9.96876085310194670618955115060089e-01 3.07272440958829056931178591582920e-03 4.62259156452221699278140642011548e-07
            9.83613419283153045391543400910450e-01 1.61180606892572138300057105197993e-02 5.59430728673389763487382164708350e-06
            9.60049667075200341415097682329360e-01 3.92956835696624678933019936266646e-02 2.10871195185260980429087923626597e-05
            9.26681682291658592731664612074383e-01 7.21168811771902978913217907575017e-02 5.15462159883483858721815729087012e-05
            8.84219981737838889301883682492189e-01 1.13882779647510132270404881182913e-01 9.99507356684321547580715505709748e-05
            8.33569402098706113690695929108188e-01 1.63703369475031224045125100019504e-01 1.67282411381404786873683399051060e-04
            7.75809417943609802570392730558524e-01 2.20516864987566257871876018725743e-01 2.52333928453970536814193748043067e-04
            7.12171060103719444356329404399730e-01 2.83112407740025717739484889534651e-01 3.51715347163449520637157341695911e-04
            6.44010658401200752898319024097873e-01 3.50155893518353300564172059239354e-01 4.60060100401231907539523202643750e-04
            5.72780927080447255228534686466446e-01 4.20218413097379894782790188401123e-01 5.70414836186469232162665754515274e-04
            5.00000000000000222044604925031308e-01 4.91806709641576300651166775423917e-01 6.74781437797298253890299069013281e-04
            4.27219072919552467215709157244419e-01 5.63395006185773317142206906282809e-01 7.64766274264282565689687665155816e-04
            3.55989341598799580168588363449089e-01 6.33457525764799411760463954124134e-01 8.32282244274132982850278317243919e-04
            2.87828939896280500132519364342443e-01 7.00501011543127383163209742633626e-01 8.70244290704067315934955306744314e-04
            2.24190582056390030896153575667995e-01 7.63096554295586870786394229071448e-01 8.73199205629697906665132034476073e-04
            1.66430597901293941820455302149639e-01 8.19910049808121765835267069633119e-01 8.37835718883419254543465370943522e-04
            1.15780018262161166209267548765638e-01 8.69730639635642899243350711913081e-01 7.63330659244753403219752030395284e-04
            7.33183177083414627794866191834444e-02 9.11496538105962761378009417967405e-01 6.51500684151915085591533216557991e-04
            3.99503329247995475625998551549856e-02 9.44317735713490646887180446356069e-01 5.06746266957098655432656375552369e-04
            1.63865807168467880750029053160688e-02 9.67495358593895904419923681416549e-01 3.35801337320351764737652588976857e-04
            3.12391468980532938104488493991084e-03 9.80540694873564766531615077838069e-01 1.47512062280936030300021455374804e-04
            9.96876085310194670618955115060089e-01 2.99911325791893433947987723797723e-03 7.14701021626589034670942648247438e-07
            9.83613419283153045391543400910450e-01 1.57319313617098159585605543497877e-02 8.64938439252969275619631339679927e-06
            9.60049667075200341415097682329360e-01 3.83543038239972725045312529346120e-02 3.26028931016108516546814222181183e-05
            9.26681682291658592731664612074383e-01 7.03892265064069849733741079944593e-02 7.96958431512759859225228376011785e-05
            8.84219981737838889301883682492189e-01 1.11154567986548338809349445455155e-01 1.54534295097172400673679604210520e-04
            8.33569402098706113690695929108188e-01 1.59781640106263722600488108582795e-01 2.58636110600886598518549641667619e-04
            7.75809417943609802570392730558524e-01 2.15234093664632780695811220539326e-01 3.90134654857276629447843108167149e-04
            7.12171060103719444356329404399730e-01 2.76330077922032069093205564058735e-01 5.43788726368796401441918497710049e-04
            6.44010658401200752898319024097873e-01 3.41767448884246982654389057643129e-01 7.11301050886541628981429852274232e-04
            5.72780927080447255228534686466446e-01 4.10151528724592318031483273443882e-01 8.81921018725282204221704862590059e-04
            5.00000000000000222044604925031308e-01 4.80024833537599948662943916133372e-01 1.04328270459740253539582255370988e-03
            4.27219072919552467215709157244419e-01 5.49898138350608189917068102658959e-01 1.18240867680624605652595082716516e-03
            3.55989341598799580168588363449089e-01 6.18282218190953081204952468397096e-01 1.28679543057547661477624245662810e-03
            2.87828939896280500132519364342443e-01 6.83719589153168327833043349528452e-01 1.34548872629024576265399648633547e-03
            2.24190582056390030896153575667995e-01 7.44815573410567699497164539934602e-01 1.35005733393530843490515369609284e-03
            1.66430597901293941820455302149639e-01 8.00268026968936618814609573746566e-01 1.29538168337638647485687837246360e-03
            1.15780018262161166209267548765638e-01 8.48895099088651905461233582173008e-01 1.18018906577897157023082641558176e-03
            7.33183177083414627794866191834444e-02 8.89660440568793453586238229036098e-01 1.00728822361774439375636802651570e-03
            3.99503329247995475625998551549856e-02 9.21695363251203159116187180188717e-01 7.83482749112687630078932787824897e-04
            1.63865807168467880750029053160688e-02 9.44317735713490646887180446356069e-01 5.19184002083117215713137237997898e-04
            3.12391468980532938104488493991084e-03 9.57050553817281435264874289714498e-01 2.28069082338073196190506020109012e-04
            9.96876085310194670618955115060089e-01 2.89487452008442731898885114105724e-03 9.51944971000512368168935961598720e-07
            9.83613419283153045391543400910450e-01 1.51851441856957879855682946868001e-02 1.15205347768774524501358275108132e-05
            9.60049667075200341415097682329360e-01 3.70212417228651322065324791310559e-02 4.34253753513749054626473522500163e-05
            9.26681682291658592731664612074383e-01 6.79427419967601753381103435458499e-02 1.06150760670299784465750447814969e-04
            8.84219981737838889301883682492189e-01 1.07291222098938363038733712073736e-01 2.05831726293127109401331908955513e-04
            8.33569402098706113690695929108188e-01 1.54228186447977588802160653358442e-01 3.44489985819948366443932208369461e-04
            7.75809417943609802570392730558524e-01 2.07753305733961807222343054490921e-01 5.19639277776832585360733407497946e-04
            7.12171060103719444356329404399730e-01 2.66725806235310003433625070101698e-01 7.24298591564082614238651025573290e-04
            6.44010658401200752898319024097873e-01 3.29888801950675214502695098417462e-01 9.47416384990979495821228351104537e-04
            5.72780927080447255228534686466446e-01 3.95896089200173828537288045481546e-01 1.17467339935301939672640525458291e-03
            5.00000000000000222044604925031308e-01 4.63340841145829074321227381005883e-01 1.38959885871298193318890135117272e-03
            4.27219072919552467215709157244419e-01 5.30785593091484875216679029108491e-01 1.57490749207458654614077975253394e-03
            3.55989341598799580168588363449089e-01 5.96792880340983100673213357367786e-01 1.71394527470365059043655442394538e-03
            2.87828939896280500132519364342443e-01 6.59955876056348644809190773230512e-01 1.79212172331143270234687925324124e-03
            2.24190582056390030896153575667995e-01 7.18928376557696924287199635728030e-01 1.79820687352192764424385629240533e-03
            1.66430597901293941820455302149639e-01 7.72453495843680948418352727458114e-01 1.72538172145023983340694773147561e-03
            1.15780018262161166209267548765638e-01 8.19390460192720215815143092186190e-01 1.57195108444250792462115740022455e-03
            7.33183177083414627794866191834444e-02 8.58738940294898500660281115415273e-01 1.34165606289274406265343841937465e-03
            3.99503329247995475625998551549856e-02 8.89660440568793564608540691551752e-01 1.04355869141761954145553037420768e-03
            1.63865807168467880750029053160688e-02 9.11496538105962872400311880483059e-01 6.91526365363398748224710299581375e-04
            3.12391468980532938104488493991084e-03 9.23786807771574181025187044724589e-01 3.03776277636082540304501842243212e-04
            9.96876085310194670618955115060089e-01 2.76222778997023497155050009155275e-03 1.16890691400809491366720622995690e-06
            9.83613419283153045391543400910450e-01 1.44893421021960368960668219528998e-02 1.41462302590970995637637344644055e-05
            9.60049667075200341415097682329360e-01 3.53248826491868866961354456179834e-02 5.33226426295092128886630999939911e-05
            9.26681682291658592731664612074383e-01 6.48295215451187567534674371927395e-02 1.30344044933943133384585122414023e-04
            8.84219981737838889301883682492189e-01 1.02375005633374704938631793993409e-01 2.52743735526418948546950593581073e-04
            8.33569402098706113690695929108188e-01 1.47161260236899682496414243360050e-01 4.23004205598425807484386229617712e-04
            7.75809417943609802570392730558524e-01 1.98233792371696804224256993620656e-01 6.38072538946356346452326580021008e-04
            7.12171060103719444356329404399730e-01 2.54504099978710762730571559586679e-01 8.89376652303492895058900646887423e-04
            6.44010658401200752898319024097873e-01 3.14772889127355570693111985747237e-01 1.16334619814900501466592430688252e-03
            5.72780927080447255228534686466446e-01 3.77755640854983365528596550575458e-01 1.44239835288167842289319686699400e-03
            5.00000000000000222044604925031308e-01 4.42109990868919222606336916214786e-01 1.70630841396239509195897898763405e-03
            4.27219072919552467215709157244419e-01 5.06464340882855745817892056948040e-01 1.93385154866072828117040671713767e-03
            3.55989341598799580168588363449089e-01 5.69447092610482985541864309197990e-01 2.10457804060558677922143644423159e-03
            2.87828939896280500132519364342443e-01 6.29715881759128182082463354163337e-01 2.20057202562994394523521179962700e-03
            2.24190582056390030896153575667995e-01 6.85986189366142196099929151387187e-01 2.20804406904685432411206136293913e-03
            1.66430597901293941820455302149639e-01 7.37058721500939206805469439132139e-01 2.11862101796353582625065392619490e-03
            1.15780018262161166209267548765638e-01 7.81844976104464128852100657240953e-01 1.93022133323122622658440938891999e-03
            7.33183177083414627794866191834444e-02 8.19390460192720215815143092186190e-01 1.64743876580168810593030048750052e-03
            3.99503329247995475625998551549856e-02 8.48895099088652016483536044688663e-01 1.28140071824660002966500549348439e-03
            1.63865807168467880750029053160688e-02 8.69730639635642899243350711913081e-01 8.49135164654102304014615665295196e-04
            3.12391468980532938104488493991084e-03 8.81457753947868627442119304760126e-01 3.73011258063853351051725182330188e-04
            9.96876085310194670618955115060089e-01 2.60399970018839344715333439239657e-03 1.36095778130327170559637269681152e-06
            9.83613419283153045391543400910450e-01 1.36593522905843027526184840780843e-02 1.64704493715507869252678146976265e-05
            9.60049667075200341415097682329360e-01 3.33013751297694601816168358254799e-02 6.20835282404545333668002826676968e-05
            9.26681682291658592731664612074383e-01 6.11159062550251722112371055573021e-02 1.51759511449142404622761337584791e-04
            8.84219981737838889301883682492189e-01 9.65106805977668652518630665326782e-02 2.94269414799572292043561283492181e-04
            8.33569402098706113690695929108188e-01 1.38731453983511721084198597964132e-01 4.92503601641969463754977542890856e-04
            7.75809417943609802570392730558524e-01 1.86878409440906084748590387789591e-01 7.42907563047372730691275677372687e-04
            7.12171060103719444356329404399730e-01 2.39925397336047036001005494654237e-01 1.03550082641868171351440253147302e-03
            6.44010658401200752898319024097873e-01 2.96741822630023133200438678613864e-01 1.35448344239112077394548983022560e-03
            5.72780927080447255228534686466446e-01 3.56116747178715054023001584937447e-01 1.67938373754862417530642204610558e-03
            5.00000000000000222044604925031308e-01 4.16784701049352890311894270780613e-01 1.98665410004520174777620233896869e-03
            4.27219072919552467215709157244419e-01 4.77452654919991170689996806686395e-01 2.25158246691402027031547738999961e-03
            3.55989341598799580168588363449089e-01 5.36827579468682647423349862947362e-01 2.45035924280821802018204280670943e-03
            2.87828939896280500132519364342443e-01 5.93644004762659105445266050082864e-01 2.56212499533443145358080350604268e-03
            2.24190582056390030896153575667995e-01 6.46690992657800167719983619463164e-01 2.57082469204134109880266478853628e-03
            1.66430597901293941820455302149639e-01 6.94837948115194392606497331144055e-01 2.46670947487454472399348581745926e-03
            1.15780018262161166209267548765638e-01 7.37058721500939206805469439132139e-01 2.24735580876239118483517209767797e-03
            7.33183177083414627794866191834444e-02 7.72453495843680948418352727458114e-01 1.91811219582104278656708995498548e-03
            3.99503329247995475625998551549856e-02 8.00268026968936729836912036262220e-01 1.49193426573677925123240228799659e-03
            1.63865807168467880750029053160688e-02 8.19910049808121876857569532148773e-01 9.88647680893289934264678286979233e-04
            3.12391468980532938104488493991084e-03 8.30965402398517682946987861214438e-01 4.34296835866100757947427801752838e-04
            9.96876085310194670618955115060089e-01 2.42356243720336485616972943546443e-03 1.52400303090964807896314121837822e-06
            9.83613419283153045391543400910450e-01 1.27128636480230167854488243506239e-02 1.84436395511477478636758881025415e-05
            9.60049667075200341415097682329360e-01 3.09938445330422106260837011859621e-02 6.95212493053328617733169214609745e-05
            9.26681682291658592731664612074383e-01 5.68810413859130517555406925112038e-02 1.69940580519978892129126513133031e-04
            8.84219981737838889301883682492189e-01 8.98232285774676758594026182436210e-02 3.29523432849693610844493685263501e-04
            8.33569402098706113690695929108188e-01 1.29118425285809773628287189239927e-01 5.51506440499217073618332118911667e-04
            7.75809417943609802570392730558524e-01 1.73929164973607164323965434959973e-01 8.31909257821130039035895098464835e-04
            7.12171060103719444356329404399730e-01 2.23300402328259706141722062966437e-01 1.15955573321334035052920796715625e-03
            6.44010658401200752898319024097873e-01 2.76179883899893341414610858919332e-01 1.51675305426760605870262210714827e-03
            5.72780927080447255228534686466446e-01 3.31440580296126774406673121120548e-01 1.88057700337594148332076571250582e-03
            5.00000000000000222044604925031308e-01 3.87904708971804734751742671505781e-01 2.22465892140947347696555702611931e-03
            4.27219072919552467215709157244419e-01 4.44368837647483194697173303211457e-01 2.52132619472884497713316420686169e-03
            3.55989341598799580168588363449089e-01 4.99629534043716239111176946607884e-01 2.74391679459823215406211183164942e-03
            2.87828939896280500132519364342443e-01 5.52509015615350151939821898849914e-01 2.86907229019247103640744001040730e-03
            2.24190582056390030896153575667995e-01 6.01880252970002693757578526856378e-01 2.87881422659304634856236582152178e-03
            1.66430597901293941820455302149639e-01 6.46690992657800056697681156947510e-01 2.76222581458950358634929500567523e-03
            1.15780018262161166209267548765638e-01 6.85986189366142085077626688871533e-01 2.51659317514352481176542752905334e-03
            7.33183177083414627794866191834444e-02 7.18928376557696813264897173212375e-01 2.14790557077878304509388485143973e-03
            3.99503329247995475625998551549856e-02 7.44815573410567588474862077418948e-01 1.67067073948721188381139057810287e-03
            1.63865807168467880750029053160688e-02 7.63096554295586870786394229071448e-01 1.10708949453254177318350937042624e-03
            3.12391468980532938104488493991084e-03 7.73385855506406483250714245514246e-01 4.86326396944210986080653347229941e-04
            9.96876085310194670618955115060089e-01 2.22476163631224336039715616664125e-03 1.65456718719834132621528902801211e-06
            9.83613419283153045391543400910450e-01 1.16700485605920629278431377429115e-02 2.00237402386450942417921161631256e-05
            9.60049667075200341415097682329360e-01 2.84514709505510610121348236134509e-02 7.54772632210458941032574231577712e-05
            9.26681682291658592731664612074383e-01 5.22151840473708481193959585198172e-02 1.84499704133767321485859458363166e-04
            8.84219981737838889301883682492189e-01 8.24551783445912345182904346074793e-02 3.57754314360272951140579111850570e-04
            8.33569402098706113690695929108188e-01 1.18527055341060325033275546502409e-01 5.98755016539503025739532571236623e-04
            7.75809417943609802570392730558524e-01 1.59662044488369320172083121178730e-01 9.03180461456031652943410303890914e-04
            7.12171060103719444356329404399730e-01 2.04983441254463910619065813989437e-01 1.25889701594448523945535267642981e-03
            6.44010658401200752898319024097873e-01 2.53525306792041993375619313155767e-01 1.64669609165811968963522016196066e-03
            5.72780927080447255228534686466446e-01 3.04253060057646074199055874487385e-01 2.04168951089841229501464781037612e-03
            5.00000000000000222044604925031308e-01 3.56085530051859555644711008426384e-01 2.41524956277491213013641768725392e-03
            4.27219072919552467215709157244419e-01 4.07918000046073481179575992427999e-01 2.73733286969383027981139733242344e-03
            3.55989341598799580168588363449089e-01 4.58645753311677228936105166212656e-01 2.97899321764133907103344611755347e-03
            2.87828939896280500132519364342443e-01 5.07187618849255561492839206039207e-01 3.11487101585291702859903395506080e-03
            2.24190582056390030896153575667995e-01 5.52509015615350262962124361365568e-01 3.12544756194977355404085450629736e-03
            1.66430597901293941820455302149639e-01 5.93644004762659105445266050082864e-01 2.99887080521361220133980118873751e-03
            1.15780018262161166209267548765638e-01 6.29715881759128182082463354163337e-01 2.73219436357316795491168015530548e-03
            7.33183177083414627794866191834444e-02 6.59955876056348644809190773230512e-01 2.33192061074174615195597759509383e-03
            3.99503329247995475625998551549856e-02 6.83719589153168438855345812044106e-01 1.81380018943794966285176784026589e-03
            1.63865807168467880750029053160688e-02 7.00501011543127494185512205149280e-01 1.20193589762889276907620317302872e-03
            3.12391468980532938104488493991084e-03 7.09946298467407199694889641250484e-01 5.27990878188740360432706921756107e-04
            9.96876085310194670618955115060089e-01 2.01183435617071281961698581142173e-03 1.74986732119761057111396312935270e-06
            9.83613419283153045391543400910450e-01 1.05531326364010271529059181716548e-02 2.11770721448221266129934708954963e-05
            9.60049667075200341415097682329360e-01 2.57284402102473594431408798755001e-02 7.98246196502783405928027615772180e-05
            9.26681682291658592731664612074383e-01 4.72177780602174024249251260698657e-02 1.95126559702289942115072118866692e-04
            8.84219981737838889301883682492189e-01 7.45635657907173926561839039095503e-02 3.78360328042365680941555572402990e-04
            8.33569402098706113690695929108188e-01 1.07183078932517772408061773603549e-01 6.33242243018090791677188189368053e-04
            7.75809417943609802570392730558524e-01 1.44381124357484263986961536829767e-01 9.55202053367344879053479900221646e-04
            7.12171060103719444356329404399730e-01 1.85364905089523301917608932853909e-01 1.33140725018522231189177507104660e-03
            6.44010658401200752898319024097873e-01 2.29260930266852680503575356851798e-01 1.74154286452131174672830482563768e-03
            5.72780927080447255228534686466446e-01 2.75133636432471739841076896482264e-01 2.15928720380503206405231608755457e-03
            5.00000000000000222044604925031308e-01 3.22005329200600209915705818275455e-01 2.55436365179771912928274879561741e-03
            4.27219072919552467215709157244419e-01 3.68877021968729124079544590131263e-01 2.89499838566731641439910838187188e-03
            3.55989341598799580168588363449089e-01 4.14749728134347850350138742214767e-01 3.15057794083705008589268992125199e-03
            2.87828939896280500132519364342443e-01 4.58645753311677450980710091243964e-01 3.29428205911424902552209204031897e-03
            2.24190582056390030896153575667995e-01 4.99629534043716572178084334154846e-01 3.30546779549849853788168196899733e-03
            1.66430597901293941820455302149639e-01 5.36827579468682980490257250494324e-01 3.17160044218126090445619702506974e-03
            1.15780018262161166209267548765638e-01 5.69447092610483318608771696744952e-01 2.88956391071223950284263537469087e-03
            7.33183177083414627794866191834444e-02 5.96792880340983433740120744914748e-01 2.46623509999234534831313858660451e-03
            3.99503329247995475625998551549856e-02 6.18282218190953414271859855944058e-01 1.91827186181169604718699250867076e-03
            1.63865807168467880750029053160688e-02 6.33457525764799744827371341671096e-01 1.27116527253056852374279994677408e-03
            3.12391468980532938104488493991084e-03 6.41998824045030058726979405037127e-01 5.58402215867315394361158897851283e-04
            9.96876085310194670618955115060089e-01 1.78931875214692441555808244402215e-03 1.80787219631420965127485255691253e-06
            9.83613419283153045391543400910450e-01 9.38592089467417921289094095982364e-03 2.18790530380106001788476088876223e-05
            9.60049667075200341415097682329360e-01 2.28827887298392318449202775809681e-02 8.24706586030347812543012109287588e-05
            9.26681682291658592731664612074383e-01 4.19953339889625951175666784820351e-02 2.01594645362473999863905937957043e-04
            8.84219981737838889301883682492189e-01 6.63165861975917175774597467352578e-02 3.90902275258199472223602510823071e-04
            8.33569402098706113690695929108188e-01 9.53282721604562510719915735535324e-02 6.54233055738497959810673343383769e-04
            7.75809417943609802570392730558524e-01 1.28412089432964254820390692657384e-01 9.86865240138997843952206601159105e-04
            7.12171060103719444356329404399730e-01 1.64862927034373940582412387811928e-01 1.37554094554646465925484832837355e-03
            6.44010658401200752898319024097873e-01 2.03903905111718269926868174479750e-01 1.79927179924856063715499221444816e-03
            5.72780927080447255228534686466446e-01 2.44702936653310582082454516239522e-01 2.23086359310057085608702820422877e-03
            5.00000000000000222044604925031308e-01 2.86390463540223516591964880717569e-01 2.63903609686258391192614780607073e-03
            4.27219072919552467215709157244419e-01 3.28077990427136756412807017113664e-01 2.99096224406342680798842614819932e-03
            3.55989341598799580168588363449089e-01 3.68877021968728791012637202584301e-01 3.25501379022378632341339077527209e-03
            2.87828939896280500132519364342443e-01 4.07918000046073370157273529912345e-01 3.40348143504578962495465965787389e-03
            2.24190582056390030896153575667995e-01 4.44368837647483083674870840695803e-01 3.41503795796579273646997876312525e-03
            1.66430597901293941820455302149639e-01 4.77452654919991004156543112912914e-01 3.27673314872415884571599242747197e-03
            1.15780018262161166209267548765638e-01 5.06464340882855523773287131916732e-01 2.98534762628421997121552777798570e-03
            7.33183177083414627794866191834444e-02 5.30785593091484653172074104077183e-01 2.54798624606513649665240350827844e-03
            3.99503329247995475625998551549856e-02 5.49898138350608078894765640143305e-01 1.98185903692844759063640402985129e-03
            1.63865807168467880750029053160688e-02 5.63395006185773095097601981251501e-01 1.31330205741276759799640583281644e-03
            3.12391468980532938104488493991084e-03 5.70991608328300315200465320231160e-01 5.76912219685232307946065599679741e-04
            9.96876085310194670618955115060089e-01 1.56195734490266534104374596125808e-03 1.82734550456727507498520465811787e-06
            9.83613419283153045391543400910450e-01 8.19329035842348077367525149838912e-03 2.21147209933909485036606507035728e-05
            9.60049667075200341415097682329360e-01 1.99751664623998119452163990672489e-02 8.33589827667032055684426250508068e-05
            9.26681682291658592731664612074383e-01 3.66591588541707452675311174061790e-02 2.03766101220534314591992242071683e-04
            8.84219981737838889301883682492189e-01 5.78900091310805553490581587539054e-02 3.95112838658890524242556230305468e-04
            8.33569402098706113690695929108188e-01 8.32152989506469847880154588892765e-02 6.61280059387159887401841373844036e-04
            7.75809417943609802570392730558524e-01 1.12095291028195154225954865978565e-01 9.97495156935462607555864522623779e-04
            7.12171060103719444356329404399730e-01 1.43914469948140361088562144686875e-01 1.39035744247691691981294592750373e-03
            6.44010658401200752898319024097873e-01 1.77994670799399706817567334837804e-01 1.81865246921474976553312341565061e-03
            5.72780927080447255228534686466446e-01 2.13609536459776427896883888024604e-01 2.25489310940573664185371249857326e-03
            5.00000000000000222044604925031308e-01 2.50000000000000000000000000000000e-01 2.66746220104735129452500075331045e-03
            4.27219072919552467215709157244419e-01 2.86390463540223849658872268264531e-01 3.02317908431943161567945210776998e-03
            3.55989341598799580168588363449089e-01 3.22005329200600376449159512048936e-01 3.29007483438068740883020169007978e-03
            2.87828939896280500132519364342443e-01 3.56085530051859888711618395973346e-01 3.44014168307292013404552477595644e-03
            2.24190582056390030896153575667995e-01 3.87904708971805123329801290310570e-01 3.45182268588348955054567746003613e-03
            1.66430597901293941820455302149639e-01 4.16784701049353223378801658327575e-01 3.31202814070325618489576413594477e-03
            1.15780018262161166209267548765638e-01 4.42109990868919666695546766277403e-01 3.01750398926591261589136294674063e-03
            7.33183177083414627794866191834444e-02 4.63340841145829518410437231068499e-01 2.57543161620543475595623839069503e-03
            3.99503329247995475625998551549856e-02 4.80024833537600448263304997453815e-01 2.00320642642805562400054597560484e-03
            1.63865807168467880750029053160688e-02 4.91806709641576800251527856744360e-01 1.32744815460123215949961306137084e-03
            3.12391468980532938104488493991084e-03 4.98438042655097557354082482561353e-01 5.83126369950828977485601800623272e-04
            9.96876085310194670618955115060089e-01 1.33459593765840409812506450748515e-03 1.80787219631420838072543167546202e-06
            9.83613419283153045391543400910450e-01 7.00065982217277105875696818770848e-03 2.18790530380105832381886638016155e-05
            9.60049667075200341415097682329360e-01 1.70675441949603608204899529710019e-02 8.24706586030347134916654305847317e-05
            9.26681682291658592731664612074383e-01 3.13229837193788468452382289797242e-02 2.01594645362473837233580065131378e-04
            8.84219981737838889301883682492189e-01 4.94634320645693029150358199785842e-02 3.90902275258199146962950765171740e-04
            8.33569402098706113690695929108188e-01 7.11023257408375936039490738949098e-02 6.54233055738497417709587100631552e-04
            7.75809417943609802570392730558524e-01 9.57784926234258732202775377118087e-02 9.86865240138996976590468612755558e-04
            7.12171060103719444356329404399730e-01 1.22966012861906587305682592159428e-01 1.37554094554646379189311033997001e-03
            6.44010658401200752898319024097873e-01 1.52085436487080893908085954535636e-01 1.79927179924855933611238523184284e-03
            5.72780927080447255228534686466446e-01 1.82516136266241996155557103520550e-01 2.23086359310056912136355222742168e-03
            5.00000000000000222044604925031308e-01 2.13609536459776150341127731735469e-01 2.63903609686258217720267182926364e-03
            4.27219072919552467215709157244419e-01 2.44702936653310554326878900610609e-01 2.99096224406342463958408117719046e-03
            3.55989341598799580168588363449089e-01 2.75133636432471462285320740193129e-01 3.25501379022378372132817681006145e-03
            2.87828939896280500132519364342443e-01 3.04253060057645907665602180713904e-01 3.40348143504578702286944569266325e-03
            2.24190582056390030896153575667995e-01 3.31440580296126663384370658604894e-01 3.41503795796579013438476479791461e-03
            1.66430597901293941820455302149639e-01 3.56116747178714887489547891163966e-01 3.27673314872415624363077846226133e-03
            1.15780018262161166209267548765638e-01 3.77755640854983143483991625544149e-01 2.98534762628421780281118280697683e-03
            7.33183177083414627794866191834444e-02 3.95896089200173662003834351708065e-01 2.54798624606513432824805853726957e-03
            3.99503329247995475625998551549856e-02 4.10151528724592151498029579670401e-01 1.98185903692844628959379704724597e-03
            1.63865807168467880750029053160688e-02 4.20218413097379728249336494627642e-01 1.31330205741276651379423334731200e-03
            3.12391468980532938104488493991084e-03 4.25884476981894077862733638539794e-01 5.76912219685231874265196605477968e-04
            9.96876085310194670618955115060089e-01 1.11208033363461742878963711689266e-03 1.74986732119760972408101587505236e-06
            9.83613419283153045391543400910450e-01 5.83344808044593265972110884831636e-03 2.11770721448221198367298928610936e-05
            9.60049667075200341415097682329360e-01 1.42218927145522557736745383749621e-02 7.98246196502783134877484494396072e-05
            9.26681682291658592731664612074383e-01 2.61005396481240846406901567888781e-02 1.95126559702289860799909182453860e-04
            8.84219981737838889301883682492189e-01 4.12164524714437041641446057838039e-02 3.78360328042365518311229699577325e-04
            8.33569402098706113690695929108188e-01 5.92475189687761624734996246388619e-02 6.33242243018090574836753692267166e-04
            7.75809417943609802570392730558524e-01 7.98094576989060028315847716839926e-02 9.55202053367344445372610906019872e-04
            7.12171060103719444356329404399730e-01 1.02464034806757392503939740890928e-01 1.33140725018522187821090607684482e-03
            6.44010658401200752898319024097873e-01 1.26728411331946705375983697194897e-01 1.74154286452131087936656683723413e-03
            5.72780927080447255228534686466446e-01 1.52085436487081115952690879566944e-01 2.15928720380503119669057809915103e-03
            5.00000000000000222044604925031308e-01 1.77994670799399706817567334837804e-01 2.55436365179771826192101080721386e-03
            4.27219072919552467215709157244419e-01 2.03903905111718519727048715139972e-01 2.89499838566731511335650139926656e-03
            3.55989341598799580168588363449089e-01 2.29260930266852763770302203738538e-01 3.15057794083704878485008293864666e-03
            2.87828939896280500132519364342443e-01 2.53525306792042215420224238187075e-01 3.29428205911424772447948505771365e-03
            2.24190582056390030896153575667995e-01 2.76179883899893618970367015208467e-01 3.30546779549849723683907498639201e-03
            1.66430597901293941820455302149639e-01 2.96741822630023410756194834902999e-01 3.17160044218125960341359004246442e-03
            1.15780018262161166209267548765638e-01 3.14772889127355848248868142036372e-01 2.88956391071223820180002839208555e-03
            7.33183177083414627794866191834444e-02 3.29888801950675492058451254706597e-01 2.46623509999234448095140059820096e-03
            3.99503329247995475625998551549856e-02 3.41767448884247371232447676447919e-01 1.91827186181169517982525452026721e-03
            1.63865807168467880750029053160688e-02 3.50155893518353689142230678044143e-01 1.27116527253056809006193095257231e-03
            3.12391468980532938104488493991084e-03 3.54877261265164944958883097569924e-01 5.58402215867315177520724400750396e-04
            9.96876085310194670618955115060089e-01 8.99153053493085912227511524719148e-04 1.65456718719834619665473574023906e-06
            9.83613419283153045391543400910450e-01 4.71653215625489081325172335823481e-03 2.00237402386451552281643184727500e-05
            9.60049667075200341415097682329360e-01 1.14988619742485403268927868225546e-02 7.54772632210461109436919202586580e-05
            9.26681682291658592731664612074383e-01 2.11031336609706146600906606636272e-02 1.84499704133767863586945701115383e-04
            8.84219981737838889301883682492189e-01 3.33248399175698137297807477352762e-02 3.57754314360274035342751597355004e-04
            8.33569402098706113690695929108188e-01 4.79035425602335473982407165749464e-02 5.98755016539504760463008548043717e-04
            7.75809417943609802570392730558524e-01 6.45285375680208772575241482627462e-02 9.03180461456034255028624269101556e-04
            7.12171060103719444356329404399730e-01 8.28454986418166866579682050542033e-02 1.25889701594448892574273912714489e-03
            6.44010658401200752898319024097873e-01 1.02464034806757253726061662746361e-01 1.64669609165812446012477909818017e-03
            5.72780927080447255228534686466446e-01 1.22966012861906601183470399973885e-01 2.04168951089841793286594473499918e-03
            5.00000000000000222044604925031308e-01 1.43914469948140194555108450913394e-01 2.41524956277491950271119058868408e-03
            4.27219072919552467215709157244419e-01 1.64862927034373940582412387811928e-01 2.73733286969383851974790822225714e-03
            3.55989341598799580168588363449089e-01 1.85364905089523135384155239080428e-01 2.97899321764134774465082600158894e-03
            2.87828939896280500132519364342443e-01 2.04983441254463827352338967102696e-01 3.11487101585292613589728283329805e-03
            2.24190582056390030896153575667995e-01 2.23300402328259650630570831708610e-01 3.12544756194978309501997237873638e-03
            1.66430597901293941820455302149639e-01 2.39925397336046952734278647767496e-01 2.99887080521362130863805006697476e-03
            1.15780018262161166209267548765638e-01 2.54504099978710651708269097071025e-01 2.73219436357317576116732205093740e-03
            7.33183177083414627794866191834444e-02 2.66725806235309892411322607586044e-01 2.33192061074175309084988150232221e-03
            3.99503329247995475625998551549856e-02 2.76330077922031958070903101543081e-01 1.81380018943795486702219577068718e-03
            1.63865807168467880750029053160688e-02 2.83112407740025662228333658276824e-01 1.20193589762889623852315512664291e-03
            3.12391468980532938104488493991084e-03 2.86929786842787415412914242551778e-01 5.27990878188741986735965650012758e-04
            9.96876085310194670618955115060089e-01 7.00352252601963874353852013143751e-04 1.52400303090964913775432528625364e-06
            9.83613419283153045391543400910450e-01 3.67371706882393565460342976791708e-03 1.84436395511477580280712551541455e-05
            9.60049667075200341415097682329360e-01 8.95648839175738550877348131962208e-03 6.95212493053329159834255457361962e-05
            9.26681682291658592731664612074383e-01 1.64372763224284006156050708113980e-02 1.69940580519979000549343761683474e-04
            8.84219981737838889301883682492189e-01 2.59567896846933550414338043310636e-02 3.29523432849693827684928182364388e-04
            8.33569402098706113690695929108188e-01 3.73121726154840849254412660229718e-02 5.51506440499217507299201113113440e-04
            7.75809417943609802570392730558524e-01 5.02614170827829914722784110381326e-02 8.31909257821130581136981341217052e-04
            7.12171060103719444356329404399730e-01 6.45285375680208495019485326338327e-02 1.15955573321334121789094595555980e-03
            6.44010658401200752898319024097873e-01 7.98094576989058640537066935394250e-02 1.51675305426760714290479459265271e-03
            5.72780927080447255228534686466446e-01 9.57784926234258593424897298973519e-02 1.88057700337594278436337269511114e-03
            5.00000000000000222044604925031308e-01 1.12095291028194959936925556576170e-01 2.22465892140947521168903300292641e-03
            4.27219072919552467215709157244419e-01 1.28412089432964227064815077028470e-01 2.52132619472884671185664018366879e-03
            3.55989341598799580168588363449089e-01 1.44381124357484097453507843056286e-01 2.74391679459823388878558780845651e-03
            2.87828939896280500132519364342443e-01 1.59662044488369181394205043034162e-01 2.86907229019247277113091598721439e-03
            2.24190582056390030896153575667995e-01 1.73929164973607053301662972444319e-01 2.87881422659304851696671079253065e-03
            1.66430597901293941820455302149639e-01 1.86878409440905945970712309645023e-01 2.76222581458950532107277098248233e-03
            1.15780018262161166209267548765638e-01 1.98233792371696665446378915476089e-01 2.51659317514352654648890350586043e-03
            7.33183177083414627794866191834444e-02 2.07753305733961640688889360717440e-01 2.14790557077878434613649183404505e-03
            3.99503329247995475625998551549856e-02 2.15234093664632669673508758023672e-01 1.67067073948721296801356306360731e-03
            1.63865807168467880750029053160688e-02 2.20516864987566119093997940581175e-01 1.10708949453254242370481286172890e-03
            3.12391468980532938104488493991084e-03 2.23490229803788076345938407030189e-01 4.86326396944211311341305092881271e-04
            9.96876085310194670618955115060089e-01 5.19914989616936150731985044615158e-04 1.36095778130327615251934578188830e-06
            9.83613419283153045391543400910450e-01 2.72722842626265272319985299986911e-03 1.64704493715508411353764389728482e-05
            9.60049667075200341415097682329360e-01 6.64895779503014722894294052935038e-03 6.20835282404547366547076236997782e-05
            9.26681682291658592731664612074383e-01 1.22024114533162975071434175333707e-02 1.51759511449142892513738956061786e-04
            8.84219981737838889301883682492189e-01 1.92693376643941968739959236245340e-02 2.94269414799573267825516520446172e-04
            8.33569402098706113690695929108188e-01 2.76991439177821791028932807421370e-02 4.92503601641971090058236271147507e-04
            7.75809417943609802570392730558524e-01 3.73121726154841196199107855591137e-02 7.42907563047375115936055145482442e-04
            7.12171060103719444356329404399730e-01 4.79035425602335820927102361110883e-02 1.03550082641868496612091998798633e-03
            6.44010658401200752898319024097873e-01 5.92475189687761277790301051027200e-02 1.35448344239112532759461426934422e-03
            5.72780927080447255228534686466446e-01 7.11023257408376491151003051527368e-02 1.67938373754862981315771897072864e-03
            5.00000000000000222044604925031308e-01 8.32152989506469292768642276314495e-02 1.98665410004520825298923725199529e-03
            4.27219072919552467215709157244419e-01 9.53282721604563204609306126258161e-02 2.25158246691402764289025029142977e-03
            3.55989341598799580168588363449089e-01 1.07183078932517758530273965789092e-01 2.45035924280822626011855369654313e-03
            2.87828939896280500132519364342443e-01 1.18527055341060366666638969945780e-01 2.56212499533444012719818339007816e-03
            2.24190582056390030896153575667995e-01 1.29118425285809829139438420497754e-01 2.57082469204134933873917567836997e-03
            1.66430597901293941820455302149639e-01 1.38731453983511776595349829221959e-01 2.46670947487455296392999670729296e-03
            1.15780018262161166209267548765638e-01 1.47161260236899738007565474617877e-01 2.24735580876239855740994499910812e-03
            7.33183177083414627794866191834444e-02 1.54228186447977644313311884616269e-01 1.91811219582104929178012486801208e-03
            3.99503329247995475625998551549856e-02 1.59781640106263805867214955469535e-01 1.49193426573678423856239572131699e-03
            1.63865807168467880750029053160688e-02 1.63703369475031307311851946906245e-01 9.88647680893293186871195743492535e-04
            3.12391468980532938104488493991084e-03 1.65910682911677015427542869474564e-01 4.34296835866102221620360657183824e-04
            9.96876085310194670618955115060089e-01 3.61686899835094572124710721183760e-04 1.16890691400809957234841612860876e-06
            9.83613419283153045391543400910450e-01 1.89723861465091836291108062795274e-03 1.41462302590971554679382532482279e-05
            9.60049667075200341415097682329360e-01 4.62545027561271637761564079482923e-03 5.33226426295094229528340190604752e-05
            9.26681682291658592731664612074383e-01 8.48879616322270949546613394431915e-03 1.30344044933943648380617053028629e-04
            8.84219981737838889301883682492189e-01 1.34050126287863606566741481174176e-02 2.52743735526419978539014454810285e-04
            8.33569402098706113690695929108188e-01 1.92693376643942072823367794853766e-02 4.23004205598427487997753582149585e-04
            7.75809417943609802570392730558524e-01 2.59567896846934001442441797280480e-02 6.38072538946358948537540545231650e-04
            7.12171060103719444356329404399730e-01 3.33248399175698553631441711786465e-02 8.89376652303496472926069849052055e-04
            6.44010658401200752898319024097873e-01 4.12164524714437041641446057838039e-02 1.16334619814900978515548324310203e-03
            5.72780927080447255228534686466446e-01 4.94634320645693723039748590508680e-02 1.44239835288168427758492828871795e-03
            5.00000000000000222044604925031308e-01 5.78900091310805553490581587539054e-02 1.70630841396240203085288289486243e-03
            4.27219072919552467215709157244419e-01 6.63165861975918147219744014364551e-02 1.93385154866073608742604861276959e-03
            3.55989341598799580168588363449089e-01 7.45635657907174204117595195384638e-02 2.10457804060559545283881632826706e-03
            2.87828939896280500132519364342443e-01 8.24551783445913177850172814942198e-02 2.20057202562995261885259168366247e-03
            2.24190582056390030896153575667995e-01 8.98232285774677868817050807592750e-02 2.20804406904686299772944124697460e-03
            1.66430597901293941820455302149639e-01 9.65106805977669623963777212338755e-02 2.11862101796354406618716481602860e-03
            1.15780018262161166209267548765638e-01 1.02375005633374802083146448694606e-01 1.93022133323123403284005128455192e-03
            7.33183177083414627794866191834444e-02 1.07291222098938460183248366774933e-01 1.64743876580169482798376989762801e-03
            3.99503329247995475625998551549856e-02 1.11154567986548449831651907970809e-01 1.28140071824660523383543342390567e-03
            1.63865807168467880750029053160688e-02 1.13882779647510257170495151513023e-01 8.49135164654105665041350370358941e-04
            3.12391468980532938104488493991084e-03 1.15418331362326070932411425928876e-01 3.73011258063854814724658037761174e-04
            9.96876085310194670618955115060089e-01 2.29040169720902116266142423128827e-04 9.51944971000513744597475249836771e-07
            9.83613419283153045391543400910450e-01 1.20143653115116683972873889985067e-03 1.15205347768774693907947725968199e-05
            9.60049667075200341415097682329360e-01 2.92909120193447173458034527016025e-03 4.34253753513749732252831325940434e-05
            9.26681682291658592731664612074383e-01 5.37557571158129698235539351003354e-03 1.06150760670299933543549164571829e-04
            8.84219981737838889301883682492189e-01 8.48879616322270082184875406028368e-03 2.05831726293127434661983654606843e-04
            8.33569402098706113690695929108188e-01 1.22024114533162923029729896029494e-02 3.44489985819948908545018451121678e-04
            7.75809417943609802570392730558524e-01 1.64372763224284144933928786258548e-02 5.19639277776833344302254147351050e-04
            7.12171060103719444356329404399730e-01 2.11031336609706215989845645708556e-02 7.24298591564083698440823511077724e-04
            6.44010658401200752898319024097873e-01 2.61005396481240603545614931135788e-02 9.47416384990980905284052582260301e-04
            5.72780927080447255228534686466446e-01 3.13229837193788607230260367941810e-02 1.17467339935302113144988123139001e-03
            5.00000000000000222044604925031308e-01 3.66591588541707175119555017772655e-02 1.38959885871298410159324632218159e-03
            4.27219072919552467215709157244419e-01 4.19953339889626089953544862964918e-02 1.57490749207458871454512472354281e-03
            3.55989341598799580168588363449089e-01 4.72177780602173746693495104409521e-02 1.71394527470365297568133389205514e-03
            2.87828939896280500132519364342443e-01 5.22151840473708481193959585198172e-02 1.79212172331143530443209321845188e-03
            2.24190582056390030896153575667995e-01 5.68810413859130586944345964184322e-02 1.79820687352193024632907025761597e-03
            1.66430597901293941820455302149639e-01 6.11159062550251722112371055573021e-02 1.72538172145024221865172719958537e-03
            1.15780018262161166209267548765638e-01 6.48295215451187567534674371927395e-02 1.57195108444251009302550237123342e-03
            7.33183177083414627794866191834444e-02 6.79427419967601753381103435458499e-02 1.34165606289274601421734889328263e-03
            3.99503329247995475625998551549856e-02 7.03892265064069988511619158089161e-02 1.04355869141762105933857185391389e-03
            1.63865807168467880750029053160688e-02 7.21168811771902978913217907575017e-02 6.91526365363399724006665536535365e-04
            3.12391468980532938104488493991084e-03 7.30892775386205589827071094077837e-02 3.03776277636083028195479460720207e-04
            9.96876085310194670618955115060089e-01 1.24801431886394824724573204832723e-04 7.14701021626572199891115969028199e-07
            9.83613419283153045391543400910450e-01 6.54649355137136264651265271652392e-04 8.64938439252948946828897236471789e-06
            9.60049667075200341415097682329360e-01 1.59602910080232493136853655357754e-03 3.26028931016100791606335262962091e-05
            9.26681682291658592731664612074383e-01 2.92909120193446783145252432234429e-03 7.96958431512741156737753001060298e-05
            8.84219981737838889301883682492189e-01 4.62545027561270596927478493398667e-03 1.54534295097168768596401777770666e-04
            8.33569402098706113690695929108188e-01 6.64895779503013595324034668010427e-03 2.58636110600880472776275098567567e-04
            7.75809417943609802570392730558524e-01 8.95648839175738030460305338920080e-03 3.90134654857267413729376981379460e-04
            7.12171060103719444356329404399730e-01 1.14988619742485299185519309617121e-02 5.43788726368783499436065920207284e-04
            6.44010658401200752898319024097873e-01 1.42218927145522228139284948156273e-02 7.11301050886524823847756326955505e-04
            5.72780927080447255228534686466446e-01 1.70675441949603434732551932029310e-02 8.81921018725261495960210389455369e-04
            5.00000000000000222044604925031308e-01 1.99751664623997633729590717166502e-02 1.04328270459737781558628988420878e-03
            4.27219072919552467215709157244419e-01 2.28827887298392110282385658592830e-02 1.18240867680621830095033519825165e-03
            3.55989341598799580168588363449089e-01 2.57284402102473108708835525249015e-02 1.28679543057544625711541286250394e-03
            2.87828939896280500132519364342443e-01 2.84514709505510193787714001700806e-02 1.34548872629021410395055990960600e-03
            2.24190582056390030896153575667995e-01 3.09938445330421724621672296962060e-02 1.35005733393527655936128262226248e-03
            1.66430597901293941820455302149639e-01 3.33013751297694116093595084748813e-02 1.29538168337635611719604877833945e-03
            1.15780018262161166209267548765638e-01 3.53248826491868381238781182673847e-02 1.18018906577894381465521078666825e-03
            7.33183177083414627794866191834444e-02 3.70212417228650836342751517804572e-02 1.00728822361772075814900784251904e-03
            3.99503329247995475625998551549856e-02 3.83543038239972239322739255840133e-02 7.83482749112669090221783285699075e-04
            1.63865807168467880750029053160688e-02 3.92956835696624123821507623688376e-02 5.19184002083104964228588151797794e-04
            3.12391468980532938104488493991084e-03 3.98255314929131520873539784588502e-02 2.28069082338067829389752216862064e-04
            9.96876085310194670618955115060089e-01 5.11902802170384254860750761206845e-05 4.62259156452230751942764422346421e-07
            9.83613419283153045391543400910450e-01 2.68520027589737742684805610338117e-04 5.59430728673400690212401745182724e-06
            9.60049667075200341415097682329360e-01 6.54649355137130518379751098478891e-04 2.10871195185265113949870524612251e-05
            9.26681682291658592731664612074383e-01 1.20143653115115469666440706220101e-03 5.15462159883493887591911220003027e-05
            8.84219981737838889301883682492189e-01 1.89723861465089732938893440916672e-03 9.99507356684341063219820244789560e-05
            8.33569402098706113690695929108188e-01 2.72722842626262410026249938255205e-03 1.67282411381408039480200855564362e-04
            7.75809417943609802570392730558524e-01 3.67371706882390096013391023177519e-03 2.52333928453975469934078557088242e-04
            7.12171060103719444356329404399730e-01 4.71653215625484484307960997284681e-03 3.51715347163456405320952624649067e-04
            6.44010658401200752898319024097873e-01 5.83344808044586760759075971805032e-03 4.60060100401240852207446208055330e-04
            5.72780927080447255228534686466446e-01 7.00065982217270427190314308063535e-03 5.70414836186480399445042355210944e-04
            5.00000000000000222044604925031308e-01 8.19329035842339056805450070442021e-03 6.74781437797311372736586143616933e-04
            4.27219072919552467215709157244419e-01 9.38592089467408727254671418904763e-03 7.64766274264297527679667965117005e-04
            3.55989341598799580168588363449089e-01 1.05531326364009143958799796791936e-02 8.32282244274149245882865599810430e-04
            2.87828939896280500132519364342443e-01 1.16700485605919432319232953432220e-02 8.70244290704084229488846080613484e-04
            2.24190582056390030896153575667995e-01 1.27128636480228884159116020668989e-02 8.73199205629714928639240056895687e-04
            1.66430597901293941820455302149639e-01 1.36593522905841639747404059335167e-02 8.37835718883435625996269902060476e-04
            1.15780018262161166209267548765638e-01 1.44893421021958894445713639242967e-02 7.63330659244768365209732330356474e-04
            7.33183177083414627794866191834444e-02 1.51851441856956335951789327509687e-02 6.51500684151927879177168545510312e-04
            3.99503329247995475625998551549856e-02 1.57319313617096598334477164371492e-02 5.06746266957108521672425993642719e-04
            1.63865807168467880750029053160688e-02 1.61180606892570507659989686999324e-02 3.35801337320358324160796126278683e-04
            3.12391468980532938104488493991084e-03 1.63353904366297479622271993093818e-02 1.47512062280938903435778541961554e-04
            9.96876085310194670618955115060089e-01 9.75884298918152725090587734646519e-06 2.00361328812390572186966932431706e-07
            9.83613419283153045391543400910450e-01 5.11902802170389472583705847696933e-05 2.42479316225441950615696745652272e-06
            9.60049667075200341415097682329360e-01 1.24801431886394987354899077658388e-04 9.13998831301523304991747675973812e-06
            9.26681682291658592731664612074383e-01 2.29040169720902116266142423128827e-04 2.23421606398043189685605203198548e-05
            8.84219981737838889301883682492189e-01 3.61686899835094246864058975532430e-04 4.33225863344752378219365551714759e-05
            8.33569402098706113690695929108188e-01 5.19914989616936042311767796064714e-04 7.25067870770959625953758220795464e-05
            7.75809417943609802570392730558524e-01 7.00352252601964416454938255895968e-04 1.09371465126865049784059269466496e-04
            7.12171060103719444356329404399730e-01 8.99153053493086237488163270370478e-04 1.52447287063449832961170571365983e-04
            6.44010658401200752898319024097873e-01 1.11208033363461634458746463138823e-03 1.99408171289474560154697524438916e-04
            5.72780927080447255228534686466446e-01 1.33459593765840474864636799878781e-03 2.47240261133552631858667503195193e-04
            5.00000000000000222044604925031308e-01 1.56195734490266404000113897865276e-03 2.92476857727698013201395044191599e-04
            4.27219072919552467215709157244419e-01 1.78931875214692506607938593532481e-03 3.31479830747993700450215737163262e-04
            3.55989341598799580168588363449089e-01 2.01183435617071195225524782301818e-03 3.60743911899038117822857696381789e-04
            2.87828939896280500132519364342443e-01 2.22476163631224336039715616664125e-03 3.77198158312490045961790086082033e-04
            2.24190582056390030896153575667995e-01 2.42356243720336528985059842966621e-03 3.78478934848255899048297123954399e-04
            1.66430597901293941820455302149639e-01 2.60399970018839344715333439239657e-03 3.63151006570309271750257584798760e-04
            1.15780018262161166209267548765638e-01 2.76222778997023497155050009155275e-03 3.30857578643387485045623819246430e-04
            7.33183177083414627794866191834444e-02 2.89487452008442731898885114105724e-03 2.82386061967279157220583885035126e-04
            3.99503329247995475625998551549856e-02 2.99911325791893477316074623217901e-03 2.19643795046679395121039246596695e-04
            1.63865807168467880750029053160688e-02 3.07272440958829056931178591582920e-03 1.45549528275136714258047865300227e-04
            3.12391468980532938104488493991084e-03 3.11415584681614798084892115070943e-03 6.39375389366014361966841783591065e-05
            ];
        
    case  40
        
        % ALG. DEG.:   40
        % PTS CARD.:  441
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.4e-15
        
        xyw=[
            9.96876085310194670618955115060089e-01 3.11415584681614798084892115070943e-03 2.00361328812385754687079423598528e-07
            9.83613419283153045391543400910450e-01 1.63353904366299144956808930828629e-02 2.42479316225436106088360690979933e-06
            9.60049667075200341415097682329360e-01 3.98255314929132075985052097166772e-02 9.13998831301501451541708515025064e-06
            9.26681682291658592731664612074383e-01 7.30892775386205589827071094077837e-02 2.23421606398037836437378556020406e-05
            8.84219981737838889301883682492189e-01 1.15418331362325959910108963413222e-01 4.33225863344741942773455378734582e-05
            8.33569402098706113690695929108188e-01 1.65910682911676959916391638216737e-01 7.25067870770942143193726892036466e-05
            7.75809417943609802570392730558524e-01 2.23490229803788242879392100803670e-01 1.09371465126862420593790992118244e-04
            7.12171060103719444356329404399730e-01 2.86929786842787526435216705067432e-01 1.52447287063446173778838432788518e-04
            6.44010658401200752898319024097873e-01 3.54877261265164611891975710022962e-01 1.99408171289469762560084276081795e-04
            5.72780927080447255228534686466446e-01 4.25884476981894299907338563571102e-01 2.47240261133546722956827457196027e-04
            5.00000000000000222044604925031308e-01 4.98438042655097113264872632498736e-01 2.92476857727690965887273888412778e-04
            4.27219072919552467215709157244419e-01 5.70991608328300537245070245262468e-01 3.31479830747985731564247968705672e-04
            3.55989341598799580168588363449089e-01 6.41998824045029725660072017490165e-01 3.60743911899029444205477812346317e-04
            2.87828939896280500132519364342443e-01 7.09946298467407199694889641250484e-01 3.77198158312480992873649832120009e-04
            2.24190582056390030896153575667995e-01 7.73385855506406594273016708029900e-01 3.78478934848246791750048245717153e-04
            1.66430597901293941820455302149639e-01 8.30965402398517682946987861214438e-01 3.63151006570300543922769076488066e-04
            1.15780018262161166209267548765638e-01 8.81457753947868627442119304760126e-01 3.30857578643379516159656050788840e-04
            7.33183177083414627794866191834444e-02 9.23786807771574181025187044724589e-01 2.82386061967272380957005850632413e-04
            3.99503329247995475625998551549856e-02 9.57050553817281546287176752230152e-01 2.19643795046674109635448379762579e-04
            1.63865807168467880750029053160688e-02 9.80540694873564877553917540353723e-01 1.45549528275133217706041599548428e-04
            3.12391468980532938104488493991084e-03 9.93761929463378490545721888338448e-01 6.39375389365999047611155425840934e-05
            9.96876085310194670618955115060089e-01 3.07272440958829056931178591582920e-03 4.62259156452221699278140642011548e-07
            9.83613419283153045391543400910450e-01 1.61180606892572138300057105197993e-02 5.59430728673389763487382164708350e-06
            9.60049667075200341415097682329360e-01 3.92956835696624678933019936266646e-02 2.10871195185260980429087923626597e-05
            9.26681682291658592731664612074383e-01 7.21168811771902978913217907575017e-02 5.15462159883483858721815729087012e-05
            8.84219981737838889301883682492189e-01 1.13882779647510132270404881182913e-01 9.99507356684321547580715505709748e-05
            8.33569402098706113690695929108188e-01 1.63703369475031224045125100019504e-01 1.67282411381404786873683399051060e-04
            7.75809417943609802570392730558524e-01 2.20516864987566257871876018725743e-01 2.52333928453970536814193748043067e-04
            7.12171060103719444356329404399730e-01 2.83112407740025717739484889534651e-01 3.51715347163449520637157341695911e-04
            6.44010658401200752898319024097873e-01 3.50155893518353300564172059239354e-01 4.60060100401231907539523202643750e-04
            5.72780927080447255228534686466446e-01 4.20218413097379894782790188401123e-01 5.70414836186469232162665754515274e-04
            5.00000000000000222044604925031308e-01 4.91806709641576300651166775423917e-01 6.74781437797298253890299069013281e-04
            4.27219072919552467215709157244419e-01 5.63395006185773317142206906282809e-01 7.64766274264282565689687665155816e-04
            3.55989341598799580168588363449089e-01 6.33457525764799411760463954124134e-01 8.32282244274132982850278317243919e-04
            2.87828939896280500132519364342443e-01 7.00501011543127383163209742633626e-01 8.70244290704067315934955306744314e-04
            2.24190582056390030896153575667995e-01 7.63096554295586870786394229071448e-01 8.73199205629697906665132034476073e-04
            1.66430597901293941820455302149639e-01 8.19910049808121765835267069633119e-01 8.37835718883419254543465370943522e-04
            1.15780018262161166209267548765638e-01 8.69730639635642899243350711913081e-01 7.63330659244753403219752030395284e-04
            7.33183177083414627794866191834444e-02 9.11496538105962761378009417967405e-01 6.51500684151915085591533216557991e-04
            3.99503329247995475625998551549856e-02 9.44317735713490646887180446356069e-01 5.06746266957098655432656375552369e-04
            1.63865807168467880750029053160688e-02 9.67495358593895904419923681416549e-01 3.35801337320351764737652588976857e-04
            3.12391468980532938104488493991084e-03 9.80540694873564766531615077838069e-01 1.47512062280936030300021455374804e-04
            9.96876085310194670618955115060089e-01 2.99911325791893433947987723797723e-03 7.14701021626589034670942648247438e-07
            9.83613419283153045391543400910450e-01 1.57319313617098159585605543497877e-02 8.64938439252969275619631339679927e-06
            9.60049667075200341415097682329360e-01 3.83543038239972725045312529346120e-02 3.26028931016108516546814222181183e-05
            9.26681682291658592731664612074383e-01 7.03892265064069849733741079944593e-02 7.96958431512759859225228376011785e-05
            8.84219981737838889301883682492189e-01 1.11154567986548338809349445455155e-01 1.54534295097172400673679604210520e-04
            8.33569402098706113690695929108188e-01 1.59781640106263722600488108582795e-01 2.58636110600886598518549641667619e-04
            7.75809417943609802570392730558524e-01 2.15234093664632780695811220539326e-01 3.90134654857276629447843108167149e-04
            7.12171060103719444356329404399730e-01 2.76330077922032069093205564058735e-01 5.43788726368796401441918497710049e-04
            6.44010658401200752898319024097873e-01 3.41767448884246982654389057643129e-01 7.11301050886541628981429852274232e-04
            5.72780927080447255228534686466446e-01 4.10151528724592318031483273443882e-01 8.81921018725282204221704862590059e-04
            5.00000000000000222044604925031308e-01 4.80024833537599948662943916133372e-01 1.04328270459740253539582255370988e-03
            4.27219072919552467215709157244419e-01 5.49898138350608189917068102658959e-01 1.18240867680624605652595082716516e-03
            3.55989341598799580168588363449089e-01 6.18282218190953081204952468397096e-01 1.28679543057547661477624245662810e-03
            2.87828939896280500132519364342443e-01 6.83719589153168327833043349528452e-01 1.34548872629024576265399648633547e-03
            2.24190582056390030896153575667995e-01 7.44815573410567699497164539934602e-01 1.35005733393530843490515369609284e-03
            1.66430597901293941820455302149639e-01 8.00268026968936618814609573746566e-01 1.29538168337638647485687837246360e-03
            1.15780018262161166209267548765638e-01 8.48895099088651905461233582173008e-01 1.18018906577897157023082641558176e-03
            7.33183177083414627794866191834444e-02 8.89660440568793453586238229036098e-01 1.00728822361774439375636802651570e-03
            3.99503329247995475625998551549856e-02 9.21695363251203159116187180188717e-01 7.83482749112687630078932787824897e-04
            1.63865807168467880750029053160688e-02 9.44317735713490646887180446356069e-01 5.19184002083117215713137237997898e-04
            3.12391468980532938104488493991084e-03 9.57050553817281435264874289714498e-01 2.28069082338073196190506020109012e-04
            9.96876085310194670618955115060089e-01 2.89487452008442731898885114105724e-03 9.51944971000512368168935961598720e-07
            9.83613419283153045391543400910450e-01 1.51851441856957879855682946868001e-02 1.15205347768774524501358275108132e-05
            9.60049667075200341415097682329360e-01 3.70212417228651322065324791310559e-02 4.34253753513749054626473522500163e-05
            9.26681682291658592731664612074383e-01 6.79427419967601753381103435458499e-02 1.06150760670299784465750447814969e-04
            8.84219981737838889301883682492189e-01 1.07291222098938363038733712073736e-01 2.05831726293127109401331908955513e-04
            8.33569402098706113690695929108188e-01 1.54228186447977588802160653358442e-01 3.44489985819948366443932208369461e-04
            7.75809417943609802570392730558524e-01 2.07753305733961807222343054490921e-01 5.19639277776832585360733407497946e-04
            7.12171060103719444356329404399730e-01 2.66725806235310003433625070101698e-01 7.24298591564082614238651025573290e-04
            6.44010658401200752898319024097873e-01 3.29888801950675214502695098417462e-01 9.47416384990979495821228351104537e-04
            5.72780927080447255228534686466446e-01 3.95896089200173828537288045481546e-01 1.17467339935301939672640525458291e-03
            5.00000000000000222044604925031308e-01 4.63340841145829074321227381005883e-01 1.38959885871298193318890135117272e-03
            4.27219072919552467215709157244419e-01 5.30785593091484875216679029108491e-01 1.57490749207458654614077975253394e-03
            3.55989341598799580168588363449089e-01 5.96792880340983100673213357367786e-01 1.71394527470365059043655442394538e-03
            2.87828939896280500132519364342443e-01 6.59955876056348644809190773230512e-01 1.79212172331143270234687925324124e-03
            2.24190582056390030896153575667995e-01 7.18928376557696924287199635728030e-01 1.79820687352192764424385629240533e-03
            1.66430597901293941820455302149639e-01 7.72453495843680948418352727458114e-01 1.72538172145023983340694773147561e-03
            1.15780018262161166209267548765638e-01 8.19390460192720215815143092186190e-01 1.57195108444250792462115740022455e-03
            7.33183177083414627794866191834444e-02 8.58738940294898500660281115415273e-01 1.34165606289274406265343841937465e-03
            3.99503329247995475625998551549856e-02 8.89660440568793564608540691551752e-01 1.04355869141761954145553037420768e-03
            1.63865807168467880750029053160688e-02 9.11496538105962872400311880483059e-01 6.91526365363398748224710299581375e-04
            3.12391468980532938104488493991084e-03 9.23786807771574181025187044724589e-01 3.03776277636082540304501842243212e-04
            9.96876085310194670618955115060089e-01 2.76222778997023497155050009155275e-03 1.16890691400809491366720622995690e-06
            9.83613419283153045391543400910450e-01 1.44893421021960368960668219528998e-02 1.41462302590970995637637344644055e-05
            9.60049667075200341415097682329360e-01 3.53248826491868866961354456179834e-02 5.33226426295092128886630999939911e-05
            9.26681682291658592731664612074383e-01 6.48295215451187567534674371927395e-02 1.30344044933943133384585122414023e-04
            8.84219981737838889301883682492189e-01 1.02375005633374704938631793993409e-01 2.52743735526418948546950593581073e-04
            8.33569402098706113690695929108188e-01 1.47161260236899682496414243360050e-01 4.23004205598425807484386229617712e-04
            7.75809417943609802570392730558524e-01 1.98233792371696804224256993620656e-01 6.38072538946356346452326580021008e-04
            7.12171060103719444356329404399730e-01 2.54504099978710762730571559586679e-01 8.89376652303492895058900646887423e-04
            6.44010658401200752898319024097873e-01 3.14772889127355570693111985747237e-01 1.16334619814900501466592430688252e-03
            5.72780927080447255228534686466446e-01 3.77755640854983365528596550575458e-01 1.44239835288167842289319686699400e-03
            5.00000000000000222044604925031308e-01 4.42109990868919222606336916214786e-01 1.70630841396239509195897898763405e-03
            4.27219072919552467215709157244419e-01 5.06464340882855745817892056948040e-01 1.93385154866072828117040671713767e-03
            3.55989341598799580168588363449089e-01 5.69447092610482985541864309197990e-01 2.10457804060558677922143644423159e-03
            2.87828939896280500132519364342443e-01 6.29715881759128182082463354163337e-01 2.20057202562994394523521179962700e-03
            2.24190582056390030896153575667995e-01 6.85986189366142196099929151387187e-01 2.20804406904685432411206136293913e-03
            1.66430597901293941820455302149639e-01 7.37058721500939206805469439132139e-01 2.11862101796353582625065392619490e-03
            1.15780018262161166209267548765638e-01 7.81844976104464128852100657240953e-01 1.93022133323122622658440938891999e-03
            7.33183177083414627794866191834444e-02 8.19390460192720215815143092186190e-01 1.64743876580168810593030048750052e-03
            3.99503329247995475625998551549856e-02 8.48895099088652016483536044688663e-01 1.28140071824660002966500549348439e-03
            1.63865807168467880750029053160688e-02 8.69730639635642899243350711913081e-01 8.49135164654102304014615665295196e-04
            3.12391468980532938104488493991084e-03 8.81457753947868627442119304760126e-01 3.73011258063853351051725182330188e-04
            9.96876085310194670618955115060089e-01 2.60399970018839344715333439239657e-03 1.36095778130327170559637269681152e-06
            9.83613419283153045391543400910450e-01 1.36593522905843027526184840780843e-02 1.64704493715507869252678146976265e-05
            9.60049667075200341415097682329360e-01 3.33013751297694601816168358254799e-02 6.20835282404545333668002826676968e-05
            9.26681682291658592731664612074383e-01 6.11159062550251722112371055573021e-02 1.51759511449142404622761337584791e-04
            8.84219981737838889301883682492189e-01 9.65106805977668652518630665326782e-02 2.94269414799572292043561283492181e-04
            8.33569402098706113690695929108188e-01 1.38731453983511721084198597964132e-01 4.92503601641969463754977542890856e-04
            7.75809417943609802570392730558524e-01 1.86878409440906084748590387789591e-01 7.42907563047372730691275677372687e-04
            7.12171060103719444356329404399730e-01 2.39925397336047036001005494654237e-01 1.03550082641868171351440253147302e-03
            6.44010658401200752898319024097873e-01 2.96741822630023133200438678613864e-01 1.35448344239112077394548983022560e-03
            5.72780927080447255228534686466446e-01 3.56116747178715054023001584937447e-01 1.67938373754862417530642204610558e-03
            5.00000000000000222044604925031308e-01 4.16784701049352890311894270780613e-01 1.98665410004520174777620233896869e-03
            4.27219072919552467215709157244419e-01 4.77452654919991170689996806686395e-01 2.25158246691402027031547738999961e-03
            3.55989341598799580168588363449089e-01 5.36827579468682647423349862947362e-01 2.45035924280821802018204280670943e-03
            2.87828939896280500132519364342443e-01 5.93644004762659105445266050082864e-01 2.56212499533443145358080350604268e-03
            2.24190582056390030896153575667995e-01 6.46690992657800167719983619463164e-01 2.57082469204134109880266478853628e-03
            1.66430597901293941820455302149639e-01 6.94837948115194392606497331144055e-01 2.46670947487454472399348581745926e-03
            1.15780018262161166209267548765638e-01 7.37058721500939206805469439132139e-01 2.24735580876239118483517209767797e-03
            7.33183177083414627794866191834444e-02 7.72453495843680948418352727458114e-01 1.91811219582104278656708995498548e-03
            3.99503329247995475625998551549856e-02 8.00268026968936729836912036262220e-01 1.49193426573677925123240228799659e-03
            1.63865807168467880750029053160688e-02 8.19910049808121876857569532148773e-01 9.88647680893289934264678286979233e-04
            3.12391468980532938104488493991084e-03 8.30965402398517682946987861214438e-01 4.34296835866100757947427801752838e-04
            9.96876085310194670618955115060089e-01 2.42356243720336485616972943546443e-03 1.52400303090964807896314121837822e-06
            9.83613419283153045391543400910450e-01 1.27128636480230167854488243506239e-02 1.84436395511477478636758881025415e-05
            9.60049667075200341415097682329360e-01 3.09938445330422106260837011859621e-02 6.95212493053328617733169214609745e-05
            9.26681682291658592731664612074383e-01 5.68810413859130517555406925112038e-02 1.69940580519978892129126513133031e-04
            8.84219981737838889301883682492189e-01 8.98232285774676758594026182436210e-02 3.29523432849693610844493685263501e-04
            8.33569402098706113690695929108188e-01 1.29118425285809773628287189239927e-01 5.51506440499217073618332118911667e-04
            7.75809417943609802570392730558524e-01 1.73929164973607164323965434959973e-01 8.31909257821130039035895098464835e-04
            7.12171060103719444356329404399730e-01 2.23300402328259706141722062966437e-01 1.15955573321334035052920796715625e-03
            6.44010658401200752898319024097873e-01 2.76179883899893341414610858919332e-01 1.51675305426760605870262210714827e-03
            5.72780927080447255228534686466446e-01 3.31440580296126774406673121120548e-01 1.88057700337594148332076571250582e-03
            5.00000000000000222044604925031308e-01 3.87904708971804734751742671505781e-01 2.22465892140947347696555702611931e-03
            4.27219072919552467215709157244419e-01 4.44368837647483194697173303211457e-01 2.52132619472884497713316420686169e-03
            3.55989341598799580168588363449089e-01 4.99629534043716239111176946607884e-01 2.74391679459823215406211183164942e-03
            2.87828939896280500132519364342443e-01 5.52509015615350151939821898849914e-01 2.86907229019247103640744001040730e-03
            2.24190582056390030896153575667995e-01 6.01880252970002693757578526856378e-01 2.87881422659304634856236582152178e-03
            1.66430597901293941820455302149639e-01 6.46690992657800056697681156947510e-01 2.76222581458950358634929500567523e-03
            1.15780018262161166209267548765638e-01 6.85986189366142085077626688871533e-01 2.51659317514352481176542752905334e-03
            7.33183177083414627794866191834444e-02 7.18928376557696813264897173212375e-01 2.14790557077878304509388485143973e-03
            3.99503329247995475625998551549856e-02 7.44815573410567588474862077418948e-01 1.67067073948721188381139057810287e-03
            1.63865807168467880750029053160688e-02 7.63096554295586870786394229071448e-01 1.10708949453254177318350937042624e-03
            3.12391468980532938104488493991084e-03 7.73385855506406483250714245514246e-01 4.86326396944210986080653347229941e-04
            9.96876085310194670618955115060089e-01 2.22476163631224336039715616664125e-03 1.65456718719834132621528902801211e-06
            9.83613419283153045391543400910450e-01 1.16700485605920629278431377429115e-02 2.00237402386450942417921161631256e-05
            9.60049667075200341415097682329360e-01 2.84514709505510610121348236134509e-02 7.54772632210458941032574231577712e-05
            9.26681682291658592731664612074383e-01 5.22151840473708481193959585198172e-02 1.84499704133767321485859458363166e-04
            8.84219981737838889301883682492189e-01 8.24551783445912345182904346074793e-02 3.57754314360272951140579111850570e-04
            8.33569402098706113690695929108188e-01 1.18527055341060325033275546502409e-01 5.98755016539503025739532571236623e-04
            7.75809417943609802570392730558524e-01 1.59662044488369320172083121178730e-01 9.03180461456031652943410303890914e-04
            7.12171060103719444356329404399730e-01 2.04983441254463910619065813989437e-01 1.25889701594448523945535267642981e-03
            6.44010658401200752898319024097873e-01 2.53525306792041993375619313155767e-01 1.64669609165811968963522016196066e-03
            5.72780927080447255228534686466446e-01 3.04253060057646074199055874487385e-01 2.04168951089841229501464781037612e-03
            5.00000000000000222044604925031308e-01 3.56085530051859555644711008426384e-01 2.41524956277491213013641768725392e-03
            4.27219072919552467215709157244419e-01 4.07918000046073481179575992427999e-01 2.73733286969383027981139733242344e-03
            3.55989341598799580168588363449089e-01 4.58645753311677228936105166212656e-01 2.97899321764133907103344611755347e-03
            2.87828939896280500132519364342443e-01 5.07187618849255561492839206039207e-01 3.11487101585291702859903395506080e-03
            2.24190582056390030896153575667995e-01 5.52509015615350262962124361365568e-01 3.12544756194977355404085450629736e-03
            1.66430597901293941820455302149639e-01 5.93644004762659105445266050082864e-01 2.99887080521361220133980118873751e-03
            1.15780018262161166209267548765638e-01 6.29715881759128182082463354163337e-01 2.73219436357316795491168015530548e-03
            7.33183177083414627794866191834444e-02 6.59955876056348644809190773230512e-01 2.33192061074174615195597759509383e-03
            3.99503329247995475625998551549856e-02 6.83719589153168438855345812044106e-01 1.81380018943794966285176784026589e-03
            1.63865807168467880750029053160688e-02 7.00501011543127494185512205149280e-01 1.20193589762889276907620317302872e-03
            3.12391468980532938104488493991084e-03 7.09946298467407199694889641250484e-01 5.27990878188740360432706921756107e-04
            9.96876085310194670618955115060089e-01 2.01183435617071281961698581142173e-03 1.74986732119761057111396312935270e-06
            9.83613419283153045391543400910450e-01 1.05531326364010271529059181716548e-02 2.11770721448221266129934708954963e-05
            9.60049667075200341415097682329360e-01 2.57284402102473594431408798755001e-02 7.98246196502783405928027615772180e-05
            9.26681682291658592731664612074383e-01 4.72177780602174024249251260698657e-02 1.95126559702289942115072118866692e-04
            8.84219981737838889301883682492189e-01 7.45635657907173926561839039095503e-02 3.78360328042365680941555572402990e-04
            8.33569402098706113690695929108188e-01 1.07183078932517772408061773603549e-01 6.33242243018090791677188189368053e-04
            7.75809417943609802570392730558524e-01 1.44381124357484263986961536829767e-01 9.55202053367344879053479900221646e-04
            7.12171060103719444356329404399730e-01 1.85364905089523301917608932853909e-01 1.33140725018522231189177507104660e-03
            6.44010658401200752898319024097873e-01 2.29260930266852680503575356851798e-01 1.74154286452131174672830482563768e-03
            5.72780927080447255228534686466446e-01 2.75133636432471739841076896482264e-01 2.15928720380503206405231608755457e-03
            5.00000000000000222044604925031308e-01 3.22005329200600209915705818275455e-01 2.55436365179771912928274879561741e-03
            4.27219072919552467215709157244419e-01 3.68877021968729124079544590131263e-01 2.89499838566731641439910838187188e-03
            3.55989341598799580168588363449089e-01 4.14749728134347850350138742214767e-01 3.15057794083705008589268992125199e-03
            2.87828939896280500132519364342443e-01 4.58645753311677450980710091243964e-01 3.29428205911424902552209204031897e-03
            2.24190582056390030896153575667995e-01 4.99629534043716572178084334154846e-01 3.30546779549849853788168196899733e-03
            1.66430597901293941820455302149639e-01 5.36827579468682980490257250494324e-01 3.17160044218126090445619702506974e-03
            1.15780018262161166209267548765638e-01 5.69447092610483318608771696744952e-01 2.88956391071223950284263537469087e-03
            7.33183177083414627794866191834444e-02 5.96792880340983433740120744914748e-01 2.46623509999234534831313858660451e-03
            3.99503329247995475625998551549856e-02 6.18282218190953414271859855944058e-01 1.91827186181169604718699250867076e-03
            1.63865807168467880750029053160688e-02 6.33457525764799744827371341671096e-01 1.27116527253056852374279994677408e-03
            3.12391468980532938104488493991084e-03 6.41998824045030058726979405037127e-01 5.58402215867315394361158897851283e-04
            9.96876085310194670618955115060089e-01 1.78931875214692441555808244402215e-03 1.80787219631420965127485255691253e-06
            9.83613419283153045391543400910450e-01 9.38592089467417921289094095982364e-03 2.18790530380106001788476088876223e-05
            9.60049667075200341415097682329360e-01 2.28827887298392318449202775809681e-02 8.24706586030347812543012109287588e-05
            9.26681682291658592731664612074383e-01 4.19953339889625951175666784820351e-02 2.01594645362473999863905937957043e-04
            8.84219981737838889301883682492189e-01 6.63165861975917175774597467352578e-02 3.90902275258199472223602510823071e-04
            8.33569402098706113690695929108188e-01 9.53282721604562510719915735535324e-02 6.54233055738497959810673343383769e-04
            7.75809417943609802570392730558524e-01 1.28412089432964254820390692657384e-01 9.86865240138997843952206601159105e-04
            7.12171060103719444356329404399730e-01 1.64862927034373940582412387811928e-01 1.37554094554646465925484832837355e-03
            6.44010658401200752898319024097873e-01 2.03903905111718269926868174479750e-01 1.79927179924856063715499221444816e-03
            5.72780927080447255228534686466446e-01 2.44702936653310582082454516239522e-01 2.23086359310057085608702820422877e-03
            5.00000000000000222044604925031308e-01 2.86390463540223516591964880717569e-01 2.63903609686258391192614780607073e-03
            4.27219072919552467215709157244419e-01 3.28077990427136756412807017113664e-01 2.99096224406342680798842614819932e-03
            3.55989341598799580168588363449089e-01 3.68877021968728791012637202584301e-01 3.25501379022378632341339077527209e-03
            2.87828939896280500132519364342443e-01 4.07918000046073370157273529912345e-01 3.40348143504578962495465965787389e-03
            2.24190582056390030896153575667995e-01 4.44368837647483083674870840695803e-01 3.41503795796579273646997876312525e-03
            1.66430597901293941820455302149639e-01 4.77452654919991004156543112912914e-01 3.27673314872415884571599242747197e-03
            1.15780018262161166209267548765638e-01 5.06464340882855523773287131916732e-01 2.98534762628421997121552777798570e-03
            7.33183177083414627794866191834444e-02 5.30785593091484653172074104077183e-01 2.54798624606513649665240350827844e-03
            3.99503329247995475625998551549856e-02 5.49898138350608078894765640143305e-01 1.98185903692844759063640402985129e-03
            1.63865807168467880750029053160688e-02 5.63395006185773095097601981251501e-01 1.31330205741276759799640583281644e-03
            3.12391468980532938104488493991084e-03 5.70991608328300315200465320231160e-01 5.76912219685232307946065599679741e-04
            9.96876085310194670618955115060089e-01 1.56195734490266534104374596125808e-03 1.82734550456727507498520465811787e-06
            9.83613419283153045391543400910450e-01 8.19329035842348077367525149838912e-03 2.21147209933909485036606507035728e-05
            9.60049667075200341415097682329360e-01 1.99751664623998119452163990672489e-02 8.33589827667032055684426250508068e-05
            9.26681682291658592731664612074383e-01 3.66591588541707452675311174061790e-02 2.03766101220534314591992242071683e-04
            8.84219981737838889301883682492189e-01 5.78900091310805553490581587539054e-02 3.95112838658890524242556230305468e-04
            8.33569402098706113690695929108188e-01 8.32152989506469847880154588892765e-02 6.61280059387159887401841373844036e-04
            7.75809417943609802570392730558524e-01 1.12095291028195154225954865978565e-01 9.97495156935462607555864522623779e-04
            7.12171060103719444356329404399730e-01 1.43914469948140361088562144686875e-01 1.39035744247691691981294592750373e-03
            6.44010658401200752898319024097873e-01 1.77994670799399706817567334837804e-01 1.81865246921474976553312341565061e-03
            5.72780927080447255228534686466446e-01 2.13609536459776427896883888024604e-01 2.25489310940573664185371249857326e-03
            5.00000000000000222044604925031308e-01 2.50000000000000000000000000000000e-01 2.66746220104735129452500075331045e-03
            4.27219072919552467215709157244419e-01 2.86390463540223849658872268264531e-01 3.02317908431943161567945210776998e-03
            3.55989341598799580168588363449089e-01 3.22005329200600376449159512048936e-01 3.29007483438068740883020169007978e-03
            2.87828939896280500132519364342443e-01 3.56085530051859888711618395973346e-01 3.44014168307292013404552477595644e-03
            2.24190582056390030896153575667995e-01 3.87904708971805123329801290310570e-01 3.45182268588348955054567746003613e-03
            1.66430597901293941820455302149639e-01 4.16784701049353223378801658327575e-01 3.31202814070325618489576413594477e-03
            1.15780018262161166209267548765638e-01 4.42109990868919666695546766277403e-01 3.01750398926591261589136294674063e-03
            7.33183177083414627794866191834444e-02 4.63340841145829518410437231068499e-01 2.57543161620543475595623839069503e-03
            3.99503329247995475625998551549856e-02 4.80024833537600448263304997453815e-01 2.00320642642805562400054597560484e-03
            1.63865807168467880750029053160688e-02 4.91806709641576800251527856744360e-01 1.32744815460123215949961306137084e-03
            3.12391468980532938104488493991084e-03 4.98438042655097557354082482561353e-01 5.83126369950828977485601800623272e-04
            9.96876085310194670618955115060089e-01 1.33459593765840409812506450748515e-03 1.80787219631420838072543167546202e-06
            9.83613419283153045391543400910450e-01 7.00065982217277105875696818770848e-03 2.18790530380105832381886638016155e-05
            9.60049667075200341415097682329360e-01 1.70675441949603608204899529710019e-02 8.24706586030347134916654305847317e-05
            9.26681682291658592731664612074383e-01 3.13229837193788468452382289797242e-02 2.01594645362473837233580065131378e-04
            8.84219981737838889301883682492189e-01 4.94634320645693029150358199785842e-02 3.90902275258199146962950765171740e-04
            8.33569402098706113690695929108188e-01 7.11023257408375936039490738949098e-02 6.54233055738497417709587100631552e-04
            7.75809417943609802570392730558524e-01 9.57784926234258732202775377118087e-02 9.86865240138996976590468612755558e-04
            7.12171060103719444356329404399730e-01 1.22966012861906587305682592159428e-01 1.37554094554646379189311033997001e-03
            6.44010658401200752898319024097873e-01 1.52085436487080893908085954535636e-01 1.79927179924855933611238523184284e-03
            5.72780927080447255228534686466446e-01 1.82516136266241996155557103520550e-01 2.23086359310056912136355222742168e-03
            5.00000000000000222044604925031308e-01 2.13609536459776150341127731735469e-01 2.63903609686258217720267182926364e-03
            4.27219072919552467215709157244419e-01 2.44702936653310554326878900610609e-01 2.99096224406342463958408117719046e-03
            3.55989341598799580168588363449089e-01 2.75133636432471462285320740193129e-01 3.25501379022378372132817681006145e-03
            2.87828939896280500132519364342443e-01 3.04253060057645907665602180713904e-01 3.40348143504578702286944569266325e-03
            2.24190582056390030896153575667995e-01 3.31440580296126663384370658604894e-01 3.41503795796579013438476479791461e-03
            1.66430597901293941820455302149639e-01 3.56116747178714887489547891163966e-01 3.27673314872415624363077846226133e-03
            1.15780018262161166209267548765638e-01 3.77755640854983143483991625544149e-01 2.98534762628421780281118280697683e-03
            7.33183177083414627794866191834444e-02 3.95896089200173662003834351708065e-01 2.54798624606513432824805853726957e-03
            3.99503329247995475625998551549856e-02 4.10151528724592151498029579670401e-01 1.98185903692844628959379704724597e-03
            1.63865807168467880750029053160688e-02 4.20218413097379728249336494627642e-01 1.31330205741276651379423334731200e-03
            3.12391468980532938104488493991084e-03 4.25884476981894077862733638539794e-01 5.76912219685231874265196605477968e-04
            9.96876085310194670618955115060089e-01 1.11208033363461742878963711689266e-03 1.74986732119760972408101587505236e-06
            9.83613419283153045391543400910450e-01 5.83344808044593265972110884831636e-03 2.11770721448221198367298928610936e-05
            9.60049667075200341415097682329360e-01 1.42218927145522557736745383749621e-02 7.98246196502783134877484494396072e-05
            9.26681682291658592731664612074383e-01 2.61005396481240846406901567888781e-02 1.95126559702289860799909182453860e-04
            8.84219981737838889301883682492189e-01 4.12164524714437041641446057838039e-02 3.78360328042365518311229699577325e-04
            8.33569402098706113690695929108188e-01 5.92475189687761624734996246388619e-02 6.33242243018090574836753692267166e-04
            7.75809417943609802570392730558524e-01 7.98094576989060028315847716839926e-02 9.55202053367344445372610906019872e-04
            7.12171060103719444356329404399730e-01 1.02464034806757392503939740890928e-01 1.33140725018522187821090607684482e-03
            6.44010658401200752898319024097873e-01 1.26728411331946705375983697194897e-01 1.74154286452131087936656683723413e-03
            5.72780927080447255228534686466446e-01 1.52085436487081115952690879566944e-01 2.15928720380503119669057809915103e-03
            5.00000000000000222044604925031308e-01 1.77994670799399706817567334837804e-01 2.55436365179771826192101080721386e-03
            4.27219072919552467215709157244419e-01 2.03903905111718519727048715139972e-01 2.89499838566731511335650139926656e-03
            3.55989341598799580168588363449089e-01 2.29260930266852763770302203738538e-01 3.15057794083704878485008293864666e-03
            2.87828939896280500132519364342443e-01 2.53525306792042215420224238187075e-01 3.29428205911424772447948505771365e-03
            2.24190582056390030896153575667995e-01 2.76179883899893618970367015208467e-01 3.30546779549849723683907498639201e-03
            1.66430597901293941820455302149639e-01 2.96741822630023410756194834902999e-01 3.17160044218125960341359004246442e-03
            1.15780018262161166209267548765638e-01 3.14772889127355848248868142036372e-01 2.88956391071223820180002839208555e-03
            7.33183177083414627794866191834444e-02 3.29888801950675492058451254706597e-01 2.46623509999234448095140059820096e-03
            3.99503329247995475625998551549856e-02 3.41767448884247371232447676447919e-01 1.91827186181169517982525452026721e-03
            1.63865807168467880750029053160688e-02 3.50155893518353689142230678044143e-01 1.27116527253056809006193095257231e-03
            3.12391468980532938104488493991084e-03 3.54877261265164944958883097569924e-01 5.58402215867315177520724400750396e-04
            9.96876085310194670618955115060089e-01 8.99153053493085912227511524719148e-04 1.65456718719834619665473574023906e-06
            9.83613419283153045391543400910450e-01 4.71653215625489081325172335823481e-03 2.00237402386451552281643184727500e-05
            9.60049667075200341415097682329360e-01 1.14988619742485403268927868225546e-02 7.54772632210461109436919202586580e-05
            9.26681682291658592731664612074383e-01 2.11031336609706146600906606636272e-02 1.84499704133767863586945701115383e-04
            8.84219981737838889301883682492189e-01 3.33248399175698137297807477352762e-02 3.57754314360274035342751597355004e-04
            8.33569402098706113690695929108188e-01 4.79035425602335473982407165749464e-02 5.98755016539504760463008548043717e-04
            7.75809417943609802570392730558524e-01 6.45285375680208772575241482627462e-02 9.03180461456034255028624269101556e-04
            7.12171060103719444356329404399730e-01 8.28454986418166866579682050542033e-02 1.25889701594448892574273912714489e-03
            6.44010658401200752898319024097873e-01 1.02464034806757253726061662746361e-01 1.64669609165812446012477909818017e-03
            5.72780927080447255228534686466446e-01 1.22966012861906601183470399973885e-01 2.04168951089841793286594473499918e-03
            5.00000000000000222044604925031308e-01 1.43914469948140194555108450913394e-01 2.41524956277491950271119058868408e-03
            4.27219072919552467215709157244419e-01 1.64862927034373940582412387811928e-01 2.73733286969383851974790822225714e-03
            3.55989341598799580168588363449089e-01 1.85364905089523135384155239080428e-01 2.97899321764134774465082600158894e-03
            2.87828939896280500132519364342443e-01 2.04983441254463827352338967102696e-01 3.11487101585292613589728283329805e-03
            2.24190582056390030896153575667995e-01 2.23300402328259650630570831708610e-01 3.12544756194978309501997237873638e-03
            1.66430597901293941820455302149639e-01 2.39925397336046952734278647767496e-01 2.99887080521362130863805006697476e-03
            1.15780018262161166209267548765638e-01 2.54504099978710651708269097071025e-01 2.73219436357317576116732205093740e-03
            7.33183177083414627794866191834444e-02 2.66725806235309892411322607586044e-01 2.33192061074175309084988150232221e-03
            3.99503329247995475625998551549856e-02 2.76330077922031958070903101543081e-01 1.81380018943795486702219577068718e-03
            1.63865807168467880750029053160688e-02 2.83112407740025662228333658276824e-01 1.20193589762889623852315512664291e-03
            3.12391468980532938104488493991084e-03 2.86929786842787415412914242551778e-01 5.27990878188741986735965650012758e-04
            9.96876085310194670618955115060089e-01 7.00352252601963874353852013143751e-04 1.52400303090964913775432528625364e-06
            9.83613419283153045391543400910450e-01 3.67371706882393565460342976791708e-03 1.84436395511477580280712551541455e-05
            9.60049667075200341415097682329360e-01 8.95648839175738550877348131962208e-03 6.95212493053329159834255457361962e-05
            9.26681682291658592731664612074383e-01 1.64372763224284006156050708113980e-02 1.69940580519979000549343761683474e-04
            8.84219981737838889301883682492189e-01 2.59567896846933550414338043310636e-02 3.29523432849693827684928182364388e-04
            8.33569402098706113690695929108188e-01 3.73121726154840849254412660229718e-02 5.51506440499217507299201113113440e-04
            7.75809417943609802570392730558524e-01 5.02614170827829914722784110381326e-02 8.31909257821130581136981341217052e-04
            7.12171060103719444356329404399730e-01 6.45285375680208495019485326338327e-02 1.15955573321334121789094595555980e-03
            6.44010658401200752898319024097873e-01 7.98094576989058640537066935394250e-02 1.51675305426760714290479459265271e-03
            5.72780927080447255228534686466446e-01 9.57784926234258593424897298973519e-02 1.88057700337594278436337269511114e-03
            5.00000000000000222044604925031308e-01 1.12095291028194959936925556576170e-01 2.22465892140947521168903300292641e-03
            4.27219072919552467215709157244419e-01 1.28412089432964227064815077028470e-01 2.52132619472884671185664018366879e-03
            3.55989341598799580168588363449089e-01 1.44381124357484097453507843056286e-01 2.74391679459823388878558780845651e-03
            2.87828939896280500132519364342443e-01 1.59662044488369181394205043034162e-01 2.86907229019247277113091598721439e-03
            2.24190582056390030896153575667995e-01 1.73929164973607053301662972444319e-01 2.87881422659304851696671079253065e-03
            1.66430597901293941820455302149639e-01 1.86878409440905945970712309645023e-01 2.76222581458950532107277098248233e-03
            1.15780018262161166209267548765638e-01 1.98233792371696665446378915476089e-01 2.51659317514352654648890350586043e-03
            7.33183177083414627794866191834444e-02 2.07753305733961640688889360717440e-01 2.14790557077878434613649183404505e-03
            3.99503329247995475625998551549856e-02 2.15234093664632669673508758023672e-01 1.67067073948721296801356306360731e-03
            1.63865807168467880750029053160688e-02 2.20516864987566119093997940581175e-01 1.10708949453254242370481286172890e-03
            3.12391468980532938104488493991084e-03 2.23490229803788076345938407030189e-01 4.86326396944211311341305092881271e-04
            9.96876085310194670618955115060089e-01 5.19914989616936150731985044615158e-04 1.36095778130327615251934578188830e-06
            9.83613419283153045391543400910450e-01 2.72722842626265272319985299986911e-03 1.64704493715508411353764389728482e-05
            9.60049667075200341415097682329360e-01 6.64895779503014722894294052935038e-03 6.20835282404547366547076236997782e-05
            9.26681682291658592731664612074383e-01 1.22024114533162975071434175333707e-02 1.51759511449142892513738956061786e-04
            8.84219981737838889301883682492189e-01 1.92693376643941968739959236245340e-02 2.94269414799573267825516520446172e-04
            8.33569402098706113690695929108188e-01 2.76991439177821791028932807421370e-02 4.92503601641971090058236271147507e-04
            7.75809417943609802570392730558524e-01 3.73121726154841196199107855591137e-02 7.42907563047375115936055145482442e-04
            7.12171060103719444356329404399730e-01 4.79035425602335820927102361110883e-02 1.03550082641868496612091998798633e-03
            6.44010658401200752898319024097873e-01 5.92475189687761277790301051027200e-02 1.35448344239112532759461426934422e-03
            5.72780927080447255228534686466446e-01 7.11023257408376491151003051527368e-02 1.67938373754862981315771897072864e-03
            5.00000000000000222044604925031308e-01 8.32152989506469292768642276314495e-02 1.98665410004520825298923725199529e-03
            4.27219072919552467215709157244419e-01 9.53282721604563204609306126258161e-02 2.25158246691402764289025029142977e-03
            3.55989341598799580168588363449089e-01 1.07183078932517758530273965789092e-01 2.45035924280822626011855369654313e-03
            2.87828939896280500132519364342443e-01 1.18527055341060366666638969945780e-01 2.56212499533444012719818339007816e-03
            2.24190582056390030896153575667995e-01 1.29118425285809829139438420497754e-01 2.57082469204134933873917567836997e-03
            1.66430597901293941820455302149639e-01 1.38731453983511776595349829221959e-01 2.46670947487455296392999670729296e-03
            1.15780018262161166209267548765638e-01 1.47161260236899738007565474617877e-01 2.24735580876239855740994499910812e-03
            7.33183177083414627794866191834444e-02 1.54228186447977644313311884616269e-01 1.91811219582104929178012486801208e-03
            3.99503329247995475625998551549856e-02 1.59781640106263805867214955469535e-01 1.49193426573678423856239572131699e-03
            1.63865807168467880750029053160688e-02 1.63703369475031307311851946906245e-01 9.88647680893293186871195743492535e-04
            3.12391468980532938104488493991084e-03 1.65910682911677015427542869474564e-01 4.34296835866102221620360657183824e-04
            9.96876085310194670618955115060089e-01 3.61686899835094572124710721183760e-04 1.16890691400809957234841612860876e-06
            9.83613419283153045391543400910450e-01 1.89723861465091836291108062795274e-03 1.41462302590971554679382532482279e-05
            9.60049667075200341415097682329360e-01 4.62545027561271637761564079482923e-03 5.33226426295094229528340190604752e-05
            9.26681682291658592731664612074383e-01 8.48879616322270949546613394431915e-03 1.30344044933943648380617053028629e-04
            8.84219981737838889301883682492189e-01 1.34050126287863606566741481174176e-02 2.52743735526419978539014454810285e-04
            8.33569402098706113690695929108188e-01 1.92693376643942072823367794853766e-02 4.23004205598427487997753582149585e-04
            7.75809417943609802570392730558524e-01 2.59567896846934001442441797280480e-02 6.38072538946358948537540545231650e-04
            7.12171060103719444356329404399730e-01 3.33248399175698553631441711786465e-02 8.89376652303496472926069849052055e-04
            6.44010658401200752898319024097873e-01 4.12164524714437041641446057838039e-02 1.16334619814900978515548324310203e-03
            5.72780927080447255228534686466446e-01 4.94634320645693723039748590508680e-02 1.44239835288168427758492828871795e-03
            5.00000000000000222044604925031308e-01 5.78900091310805553490581587539054e-02 1.70630841396240203085288289486243e-03
            4.27219072919552467215709157244419e-01 6.63165861975918147219744014364551e-02 1.93385154866073608742604861276959e-03
            3.55989341598799580168588363449089e-01 7.45635657907174204117595195384638e-02 2.10457804060559545283881632826706e-03
            2.87828939896280500132519364342443e-01 8.24551783445913177850172814942198e-02 2.20057202562995261885259168366247e-03
            2.24190582056390030896153575667995e-01 8.98232285774677868817050807592750e-02 2.20804406904686299772944124697460e-03
            1.66430597901293941820455302149639e-01 9.65106805977669623963777212338755e-02 2.11862101796354406618716481602860e-03
            1.15780018262161166209267548765638e-01 1.02375005633374802083146448694606e-01 1.93022133323123403284005128455192e-03
            7.33183177083414627794866191834444e-02 1.07291222098938460183248366774933e-01 1.64743876580169482798376989762801e-03
            3.99503329247995475625998551549856e-02 1.11154567986548449831651907970809e-01 1.28140071824660523383543342390567e-03
            1.63865807168467880750029053160688e-02 1.13882779647510257170495151513023e-01 8.49135164654105665041350370358941e-04
            3.12391468980532938104488493991084e-03 1.15418331362326070932411425928876e-01 3.73011258063854814724658037761174e-04
            9.96876085310194670618955115060089e-01 2.29040169720902116266142423128827e-04 9.51944971000513744597475249836771e-07
            9.83613419283153045391543400910450e-01 1.20143653115116683972873889985067e-03 1.15205347768774693907947725968199e-05
            9.60049667075200341415097682329360e-01 2.92909120193447173458034527016025e-03 4.34253753513749732252831325940434e-05
            9.26681682291658592731664612074383e-01 5.37557571158129698235539351003354e-03 1.06150760670299933543549164571829e-04
            8.84219981737838889301883682492189e-01 8.48879616322270082184875406028368e-03 2.05831726293127434661983654606843e-04
            8.33569402098706113690695929108188e-01 1.22024114533162923029729896029494e-02 3.44489985819948908545018451121678e-04
            7.75809417943609802570392730558524e-01 1.64372763224284144933928786258548e-02 5.19639277776833344302254147351050e-04
            7.12171060103719444356329404399730e-01 2.11031336609706215989845645708556e-02 7.24298591564083698440823511077724e-04
            6.44010658401200752898319024097873e-01 2.61005396481240603545614931135788e-02 9.47416384990980905284052582260301e-04
            5.72780927080447255228534686466446e-01 3.13229837193788607230260367941810e-02 1.17467339935302113144988123139001e-03
            5.00000000000000222044604925031308e-01 3.66591588541707175119555017772655e-02 1.38959885871298410159324632218159e-03
            4.27219072919552467215709157244419e-01 4.19953339889626089953544862964918e-02 1.57490749207458871454512472354281e-03
            3.55989341598799580168588363449089e-01 4.72177780602173746693495104409521e-02 1.71394527470365297568133389205514e-03
            2.87828939896280500132519364342443e-01 5.22151840473708481193959585198172e-02 1.79212172331143530443209321845188e-03
            2.24190582056390030896153575667995e-01 5.68810413859130586944345964184322e-02 1.79820687352193024632907025761597e-03
            1.66430597901293941820455302149639e-01 6.11159062550251722112371055573021e-02 1.72538172145024221865172719958537e-03
            1.15780018262161166209267548765638e-01 6.48295215451187567534674371927395e-02 1.57195108444251009302550237123342e-03
            7.33183177083414627794866191834444e-02 6.79427419967601753381103435458499e-02 1.34165606289274601421734889328263e-03
            3.99503329247995475625998551549856e-02 7.03892265064069988511619158089161e-02 1.04355869141762105933857185391389e-03
            1.63865807168467880750029053160688e-02 7.21168811771902978913217907575017e-02 6.91526365363399724006665536535365e-04
            3.12391468980532938104488493991084e-03 7.30892775386205589827071094077837e-02 3.03776277636083028195479460720207e-04
            9.96876085310194670618955115060089e-01 1.24801431886394824724573204832723e-04 7.14701021626572199891115969028199e-07
            9.83613419283153045391543400910450e-01 6.54649355137136264651265271652392e-04 8.64938439252948946828897236471789e-06
            9.60049667075200341415097682329360e-01 1.59602910080232493136853655357754e-03 3.26028931016100791606335262962091e-05
            9.26681682291658592731664612074383e-01 2.92909120193446783145252432234429e-03 7.96958431512741156737753001060298e-05
            8.84219981737838889301883682492189e-01 4.62545027561270596927478493398667e-03 1.54534295097168768596401777770666e-04
            8.33569402098706113690695929108188e-01 6.64895779503013595324034668010427e-03 2.58636110600880472776275098567567e-04
            7.75809417943609802570392730558524e-01 8.95648839175738030460305338920080e-03 3.90134654857267413729376981379460e-04
            7.12171060103719444356329404399730e-01 1.14988619742485299185519309617121e-02 5.43788726368783499436065920207284e-04
            6.44010658401200752898319024097873e-01 1.42218927145522228139284948156273e-02 7.11301050886524823847756326955505e-04
            5.72780927080447255228534686466446e-01 1.70675441949603434732551932029310e-02 8.81921018725261495960210389455369e-04
            5.00000000000000222044604925031308e-01 1.99751664623997633729590717166502e-02 1.04328270459737781558628988420878e-03
            4.27219072919552467215709157244419e-01 2.28827887298392110282385658592830e-02 1.18240867680621830095033519825165e-03
            3.55989341598799580168588363449089e-01 2.57284402102473108708835525249015e-02 1.28679543057544625711541286250394e-03
            2.87828939896280500132519364342443e-01 2.84514709505510193787714001700806e-02 1.34548872629021410395055990960600e-03
            2.24190582056390030896153575667995e-01 3.09938445330421724621672296962060e-02 1.35005733393527655936128262226248e-03
            1.66430597901293941820455302149639e-01 3.33013751297694116093595084748813e-02 1.29538168337635611719604877833945e-03
            1.15780018262161166209267548765638e-01 3.53248826491868381238781182673847e-02 1.18018906577894381465521078666825e-03
            7.33183177083414627794866191834444e-02 3.70212417228650836342751517804572e-02 1.00728822361772075814900784251904e-03
            3.99503329247995475625998551549856e-02 3.83543038239972239322739255840133e-02 7.83482749112669090221783285699075e-04
            1.63865807168467880750029053160688e-02 3.92956835696624123821507623688376e-02 5.19184002083104964228588151797794e-04
            3.12391468980532938104488493991084e-03 3.98255314929131520873539784588502e-02 2.28069082338067829389752216862064e-04
            9.96876085310194670618955115060089e-01 5.11902802170384254860750761206845e-05 4.62259156452230751942764422346421e-07
            9.83613419283153045391543400910450e-01 2.68520027589737742684805610338117e-04 5.59430728673400690212401745182724e-06
            9.60049667075200341415097682329360e-01 6.54649355137130518379751098478891e-04 2.10871195185265113949870524612251e-05
            9.26681682291658592731664612074383e-01 1.20143653115115469666440706220101e-03 5.15462159883493887591911220003027e-05
            8.84219981737838889301883682492189e-01 1.89723861465089732938893440916672e-03 9.99507356684341063219820244789560e-05
            8.33569402098706113690695929108188e-01 2.72722842626262410026249938255205e-03 1.67282411381408039480200855564362e-04
            7.75809417943609802570392730558524e-01 3.67371706882390096013391023177519e-03 2.52333928453975469934078557088242e-04
            7.12171060103719444356329404399730e-01 4.71653215625484484307960997284681e-03 3.51715347163456405320952624649067e-04
            6.44010658401200752898319024097873e-01 5.83344808044586760759075971805032e-03 4.60060100401240852207446208055330e-04
            5.72780927080447255228534686466446e-01 7.00065982217270427190314308063535e-03 5.70414836186480399445042355210944e-04
            5.00000000000000222044604925031308e-01 8.19329035842339056805450070442021e-03 6.74781437797311372736586143616933e-04
            4.27219072919552467215709157244419e-01 9.38592089467408727254671418904763e-03 7.64766274264297527679667965117005e-04
            3.55989341598799580168588363449089e-01 1.05531326364009143958799796791936e-02 8.32282244274149245882865599810430e-04
            2.87828939896280500132519364342443e-01 1.16700485605919432319232953432220e-02 8.70244290704084229488846080613484e-04
            2.24190582056390030896153575667995e-01 1.27128636480228884159116020668989e-02 8.73199205629714928639240056895687e-04
            1.66430597901293941820455302149639e-01 1.36593522905841639747404059335167e-02 8.37835718883435625996269902060476e-04
            1.15780018262161166209267548765638e-01 1.44893421021958894445713639242967e-02 7.63330659244768365209732330356474e-04
            7.33183177083414627794866191834444e-02 1.51851441856956335951789327509687e-02 6.51500684151927879177168545510312e-04
            3.99503329247995475625998551549856e-02 1.57319313617096598334477164371492e-02 5.06746266957108521672425993642719e-04
            1.63865807168467880750029053160688e-02 1.61180606892570507659989686999324e-02 3.35801337320358324160796126278683e-04
            3.12391468980532938104488493991084e-03 1.63353904366297479622271993093818e-02 1.47512062280938903435778541961554e-04
            9.96876085310194670618955115060089e-01 9.75884298918152725090587734646519e-06 2.00361328812390572186966932431706e-07
            9.83613419283153045391543400910450e-01 5.11902802170389472583705847696933e-05 2.42479316225441950615696745652272e-06
            9.60049667075200341415097682329360e-01 1.24801431886394987354899077658388e-04 9.13998831301523304991747675973812e-06
            9.26681682291658592731664612074383e-01 2.29040169720902116266142423128827e-04 2.23421606398043189685605203198548e-05
            8.84219981737838889301883682492189e-01 3.61686899835094246864058975532430e-04 4.33225863344752378219365551714759e-05
            8.33569402098706113690695929108188e-01 5.19914989616936042311767796064714e-04 7.25067870770959625953758220795464e-05
            7.75809417943609802570392730558524e-01 7.00352252601964416454938255895968e-04 1.09371465126865049784059269466496e-04
            7.12171060103719444356329404399730e-01 8.99153053493086237488163270370478e-04 1.52447287063449832961170571365983e-04
            6.44010658401200752898319024097873e-01 1.11208033363461634458746463138823e-03 1.99408171289474560154697524438916e-04
            5.72780927080447255228534686466446e-01 1.33459593765840474864636799878781e-03 2.47240261133552631858667503195193e-04
            5.00000000000000222044604925031308e-01 1.56195734490266404000113897865276e-03 2.92476857727698013201395044191599e-04
            4.27219072919552467215709157244419e-01 1.78931875214692506607938593532481e-03 3.31479830747993700450215737163262e-04
            3.55989341598799580168588363449089e-01 2.01183435617071195225524782301818e-03 3.60743911899038117822857696381789e-04
            2.87828939896280500132519364342443e-01 2.22476163631224336039715616664125e-03 3.77198158312490045961790086082033e-04
            2.24190582056390030896153575667995e-01 2.42356243720336528985059842966621e-03 3.78478934848255899048297123954399e-04
            1.66430597901293941820455302149639e-01 2.60399970018839344715333439239657e-03 3.63151006570309271750257584798760e-04
            1.15780018262161166209267548765638e-01 2.76222778997023497155050009155275e-03 3.30857578643387485045623819246430e-04
            7.33183177083414627794866191834444e-02 2.89487452008442731898885114105724e-03 2.82386061967279157220583885035126e-04
            3.99503329247995475625998551549856e-02 2.99911325791893477316074623217901e-03 2.19643795046679395121039246596695e-04
            1.63865807168467880750029053160688e-02 3.07272440958829056931178591582920e-03 1.45549528275136714258047865300227e-04
            3.12391468980532938104488493991084e-03 3.11415584681614798084892115070943e-03 6.39375389366014361966841783591065e-05
            ];
        
    case  41
        
        % ALG. DEG.:   41
        % PTS CARD.:  484
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.9e-15
        
        xyw=[
            9.97147292741199509080729512788821e-01 2.84456932009607803280015758673471e-03 1.52604324215667471116133547572036e-07
            9.85030248917714068568329821573570e-01 1.49270467647111132469817107448762e-02 1.84898544265612625098259367423248e-06
            9.63478386093586713911918195663020e-01 3.64174284333192987972438459109981e-02 6.98427165067680691450330635050570e-06
            9.32906288860150034736307134153321e-01 6.69023124230615079399697719964024e-02 1.71253747565580234884889676560960e-05
            8.93908402989603945165697496122448e-01 1.05788948741506766415554352533945e-01 3.33433532827636684909601572091020e-05
            8.47243631593341484098402816016460e-01 1.52320599205676837639700238469231e-01 5.60943394228683063496010419157756e-05
            7.93820201753455689797078775882255e-01 2.05591627639468366162134316255106e-01 8.51508699150972013730900855854600e-05
            7.34677918993378642475988726801006e-01 2.64565194780213730485485257304390e-01 1.19589629032227804869559228961862e-04
            6.70967910446042203886918287025765e-01 3.28093457323708981299148490506923e-01 1.57837138480472694490250318288815e-04
            6.03930213344110788931118349864846e-01 3.94939915500504390433178514285828e-01 1.97770357903992194826409134478240e-04
            5.34869636659861091843026770220604e-01 4.63803482576349956900685356231406e-01 2.36864816679008667816078159695792e-04
            4.65130363340138908156973229779396e-01 5.33343810164849552180044156557415e-01 2.72379118672929992038544977361880e-04
            3.96069786655889322091184112650808e-01 6.02207377240694952114097304729512e-01 3.01561741052138159585593868428077e-04
            3.29032089553957685090779250458581e-01 6.69053835417490638803883484797552e-01 3.21864214340280110123598822369217e-04
            2.65322081006621690590918660745956e-01 7.32582097960985501039488099195296e-01 3.31144168089031693067575412570136e-04
            2.06179798246544199180618761602091e-01 7.91555665101731253940897659049369e-01 3.27842404107201866765575193340965e-04
            1.52756368406658793457353340272675e-01 8.44826693535522310618546271143714e-01 3.11120068774749800391554721556986e-04
            1.06091597010395888300848810104071e-01 8.91358343999692825931902007141616e-01 2.80944999634538650733206477383419e-04
            6.70937111398499097525416345888516e-02 9.30244980318138070529698779864702e-01 2.38120228230910709575629535805774e-04
            3.65216139064131750657793418213259e-02 9.60729864307880321305788129393477e-01 1.84252393535425692920604046065591e-04
            1.49697510822856538759140221372945e-02 9.82220245976488715022867381776450e-01 1.21665789952908013227955708046579e-04
            2.85270725880037989696802469552495e-03 9.94302723421103595846659572998760e-01 5.33419572873506089220711767762140e-05
            9.97147292741199509080729512788821e-01 2.81000294122561722343078827179852e-03 3.52351496340051358104072422305908e-07
            9.85030248917714068568329821573570e-01 1.47456576348203860504781914642081e-02 4.26916334631605038767718576786514e-06
            9.63478386093586713911918195663020e-01 3.59748944371108755890809050015378e-02 1.61261391484784531117485650852927e-05
            9.32906288860150034736307134153321e-01 6.60893349848996752848862001883390e-02 3.95411562016396216589185497181091e-05
            8.93908402989603945165697496122448e-01 1.04503432211228236559108495384862e-01 7.69872051959229295337722165371019e-05
            8.47243631593341484098402816016460e-01 1.50469643595376872946545177001099e-01 1.29517459832407367354908944712122e-04
            7.93820201753455689797078775882255e-01 2.03093337988597605336948959120491e-01 1.96606725159634971016736626481247e-04
            7.34677918993378642475988726801006e-01 2.61350275497318151884940107265720e-01 2.76123137092146471709835919838838e-04
            6.70967910446042203886918287025765e-01 3.24106561075250709791362169198692e-01 3.64433656827639910948807289159390e-04
            6.03930213344110788931118349864846e-01 3.90140720538436458486586388971773e-01 4.56636349574849221156364054152732e-04
            5.34869636659861091843026770220604e-01 4.58167477580123827518576717920951e-01 5.46902409326301955014293376677870e-04
            4.65130363340138908156973229779396e-01 5.26862771337590296560904334910447e-01 6.28902165973734624156155081919906e-04
            3.96069786655889322091184112650808e-01 5.94889528379277554570592201343970e-01 6.96282567645111253283596308705228e-04
            3.29032089553957685090779250458581e-01 6.60923687842463580821572577406187e-01 7.43159396851935933996091687703256e-04
            2.65322081006621690590918660745956e-01 7.23679973420395583616482326760888e-01 7.64586087125260389918413039822553e-04
            2.06179798246544199180618761602091e-01 7.81936910929116546498107709339820e-01 7.56962571307220519721548068758921e-04
            1.52756368406658793457353340272675e-01 8.34560605322336890310452872654423e-01 7.18352001738021737030359670228563e-04
            1.06091597010395888300848810104071e-01 8.80526816706485915275948173075449e-01 6.48680117809674128617758448456243e-04
            6.70937111398499097525416345888516e-02 9.18940913932814407161231429199688e-01 5.49801056799817389197926331689814e-04
            3.65216139064131750657793418213259e-02 9.49055354480603297062657475180458e-01 4.25424422932426478036194161802541e-04
            1.49697510822856538759140221372945e-02 9.70284591282893904562456555140670e-01 2.80916830919659282501249508712249e-04
            2.85270725880037989696802469552495e-03 9.82220245976488604000564919260796e-01 1.23162423899226901964285008084232e-04
            9.97147292741199509080729512788821e-01 2.74852178570655697117941507201522e-03 5.45542222925190226048201922598446e-07
            9.85030248917714068568329821573570e-01 1.44230316129836250160778376994131e-02 6.60990200459456978644224092733772e-06
            9.63478386093586713911918195663020e-01 3.51877856240841099810801040348451e-02 2.49679365339532700624868705174819e-05
            9.32906288860150034736307134153321e-01 6.46433405260520010937597135125543e-02 6.12211682803705432593527335782824e-05
            8.93908402989603945165697496122448e-01 1.02216960665667586094507157667977e-01 1.19198503470659144826832021024643e-04
            8.47243631593341484098402816016460e-01 1.47177459297964718931694960701861e-01 2.00530560189263011572347039646047e-04
            7.93820201753455689797078775882255e-01 1.98649779259681830145822800659516e-01 3.04404184457091509137527385320254e-04
            7.34677918993378642475988726801006e-01 2.55632090403251410748453054111451e-01 4.27518632885124348370214919512478e-04
            6.70967910446042203886918287025765e-01 3.17015306616447778420564418411232e-01 5.64248908602996276581698076313387e-04
            6.03930213344110788931118349864846e-01 3.81604678827647370020770267728949e-01 7.07005396040916449414137634477129e-04
            5.34869636659861091843026770220604e-01 4.48143051794080649052887110883603e-01 8.46763414391945830675401118980972e-04
            4.65130363340138908156973229779396e-01 5.15335334299506064859031084779417e-01 9.73722800077631424169044560557040e-04
            3.96069786655889322091184112650808e-01 5.81873707265939232868845465418417e-01 1.07804718777348897823520346150872e-03
            3.29032089553957685090779250458581e-01 6.46463079477139102024807471025269e-01 1.15062610364247402267023900890308e-03
            2.65322081006621690590918660745956e-01 7.07846295690334970096557754004607e-01 1.18380082934410103302769190491972e-03
            2.06179798246544199180618761602091e-01 7.64828606833904967032822241890244e-01 1.17199741766832064501779875342891e-03
            1.52756368406658793457353340272675e-01 8.16300926795621717424467078672023e-01 1.11221706716610490607577421684482e-03
            1.06091597010395888300848810104071e-01 8.61261425427919280473076923954068e-01 1.00434480089658852847156467191780e-03
            6.70937111398499097525416345888516e-02 8.98835045567534796084885329037206e-01 8.51251360668898527164771827813183e-04
            3.65216139064131750657793418213259e-02 9.28290600469502624747519803349860e-01 6.58680288813750613834130387402865e-04
            1.49697510822856538759140221372945e-02 9.49055354480603408084959937696112e-01 4.34940660076290968542173098754233e-04
            2.85270725880037989696802469552495e-03 9.60729864307880210283485666877823e-01 1.90691122963173274873494023040621e-04
            9.97147292741199509080729512788821e-01 2.66130854201197765071529310887399e-03 7.28140984419258521665048976279078e-07
            9.85030248917714068568329821573570e-01 1.39653749273356345139696088608616e-02 8.82230623091535732540040964932970e-06
            9.63478386093586713911918195663020e-01 3.40712432926152108958461894872016e-02 3.33249694024935889892155560509934e-05
            9.32906288860150034736307134153321e-01 6.25921450653323940827021942823194e-02 8.17125418816189283324064041025281e-05
            8.93908402989603945165697496122448e-01 9.89735180462151670210957377094019e-02 1.59095505372697893723549222855240e-04
            8.47243631593341484098402816016460e-01 1.42507376750009656651840828089917e-01 2.67650263107825864359462508801357e-04
            7.93820201753455689797078775882255e-01 1.92346430420118119641870180203114e-01 4.06291489856547229000266474585601e-04
            7.34677918993378642475988726801006e-01 2.47520637944539240171692995318153e-01 5.70613648449420766441131380730667e-04
            6.70967910446042203886918287025765e-01 3.06956105581683347249821736113518e-01 7.53108995972283866947305508432464e-04
            6.03930213344110788931118349864846e-01 3.69495994798776983270016671667690e-01 9.43647592119656406645100688024286e-04
            5.34869636659861091843026770220604e-01 4.33923041099822159871735038905172e-01 1.13018409981093751609515507539072e-03
            4.65130363340138908156973229779396e-01 4.98983247760327874864572095248150e-01 1.29963813689491281856691617946353e-03
            3.96069786655889322091184112650808e-01 5.63410294061372995955139231227804e-01 1.43888100236641796984826680017022e-03
            3.29032089553957685090779250458581e-01 6.25950183278466854019939091813285e-01 1.53575285027867801564138705572304e-03
            2.65322081006621690590918660745956e-01 6.85385650915610544764433598174946e-01 1.58003150812609065301028721250987e-03
            2.06179798246544199180618761602091e-01 7.40559858440032026116739416465862e-01 1.56427736951693851716327365863890e-03
            1.52756368406658793457353340272675e-01 7.90398912110140017261983302887529e-01 1.48448790238786413123173169026359e-03
            1.06091597010395888300848810104071e-01 8.33932770813935020370877282402944e-01 1.34050964579782746870229015456744e-03
            6.70937111398499097525416345888516e-02 8.70314143794817751675907402386656e-01 1.13617420925214425257709915939586e-03
            3.65216139064131750657793418213259e-02 8.98835045567534907107187791552860e-01 8.79147559546779072854605363573910e-04
            1.49697510822856538759140221372945e-02 9.18940913932814629205836354230996e-01 5.80519906770517982348700503081318e-04
            2.85270725880037989696802469552495e-03 9.30244980318138181552001242380356e-01 2.54517462003045090088160229413461e-04
            9.97147292741199509080729512788821e-01 2.55005898991119773003277693135260e-03 8.96572665723641283570073332032102e-07
            9.85030248917714068568329821573570e-01 1.33815862831181617287334972843382e-02 1.08630591939426408468844995058866e-05
            9.63478386093586713911918195663020e-01 3.26469775616847618371352268695773e-02 4.10336147692364587038181544809134e-05
            9.32906288860150034736307134153321e-01 5.99756321756691329616373309363553e-02 1.00614074836466636472288860293389e-04
            8.93908402989603945165697496122448e-01 9.48361700541797830732448915114219e-02 1.95897064454372365994647142706242e-04
            8.47243631593341484098402816016460e-01 1.36550201328887704788073165218520e-01 3.29562426797892641175402594910793e-04
            7.93820201753455689797078775882255e-01 1.84305854179287159633204851161281e-01 5.00273782023194437014457225387787e-04
            7.34677918993378642475988726801006e-01 2.37173637710507229092726788621803e-01 7.02606515545368699032779957747152e-04
            6.70967910446042203886918287025765e-01 2.94124549705510829333832134580007e-01 9.27316212859322260009253824364350e-04
            6.03930213344110788931118349864846e-01 3.54050110461999079092265674262308e-01 1.16192970217875667614249035608509e-03
            5.34869636659861091843026770220604e-01 4.15783940275357810634204724919982e-01 1.39161535033512063185001661480555e-03
            4.65130363340138908156973229779396e-01 4.78124462714246134531492771202466e-01 1.60026705515185362245500488853622e-03
            3.96069786655889322091184112650808e-01 5.39858292527604755051129359344486e-01 1.77171921860664813297425723703782e-03
            3.29032089553957685090779250458581e-01 5.99783853284093337876470286573749e-01 1.89099921077127129144523909332065e-03
            2.65322081006621690590918660745956e-01 6.56734765279096466272790166840423e-01 1.94552029274632659085197960990854e-03
            2.06179798246544199180618761602091e-01 7.09602548810316924310370723105734e-01 1.92612194771256536526826685928881e-03
            1.52756368406658793457353340272675e-01 7.57358201660715990577443790243706e-01 1.82787578828557122252729261191462e-03
            1.06091597010395888300848810104071e-01 7.99072232935424286992542874941137e-01 1.65059285533800852510677437123832e-03
            6.70937111398499097525416345888516e-02 8.33932770813934909348574819887290e-01 1.39899107633406649155161094455480e-03
            3.65216139064131750657793418213259e-02 8.61261425427919280473076923954068e-01 1.08250968959802142642379063630642e-03
            1.49697510822856538759140221372945e-02 8.80526816706486026298250635591103e-01 7.14804263811629099489852734450324e-04
            2.85270725880037989696802469552495e-03 8.91358343999692825931902007141616e-01 3.13391780251574933631453223625840e-04
            9.97147292741199509080729512788821e-01 2.41693805781881439209701767367733e-03 1.04755520780146932007068517195281e-06
            9.85030248917714068568329821573570e-01 1.26830262710043335794285468409726e-02 1.26923947899811388718375321227327e-05
            9.63478386093586713911918195663020e-01 3.09427047977194266681344458902458e-02 4.79436620028326771891097335398513e-05
            9.32906288860150034736307134153321e-01 5.68447194832011615250877412108821e-02 1.17557451952874308194127439541887e-04
            8.93908402989603945165697496122448e-01 8.98854299326252437207784851125325e-02 2.28886065689462526659414542962168e-04
            8.47243631593341484098402816016460e-01 1.29421860317867731815510978776729e-01 3.85060631096953182462422349274789e-04
            7.93820201753455689797078775882255e-01 1.74684521027584666041931882318750e-01 5.84519722405268558548807611430220e-04
            7.34677918993378642475988726801006e-01 2.24792443453952606180479278918938e-01 8.20925221717237686876178504036261e-04
            6.70967910446042203886918287025765e-01 2.78770342464440823437854533040081e-01 1.08347595816505400286022808842290e-03
            6.03930213344110788931118349864846e-01 3.35567604410735575992674739609356e-01 1.35759828193529933325345293582131e-03
            5.34869636659861091843026770220604e-01 3.94078738200629719035106290903059e-01 1.62596291771109060717015459829327e-03
            4.65130363340138908156973229779396e-01 4.53164893392711765063296525113401e-01 1.86975150100568377961596233660657e-03
            3.96069786655889322091184112650808e-01 5.11676027182605852594576845149277e-01 2.07007615240573008413837996499751e-03
            3.29032089553957685090779250458581e-01 5.68473289128900827194001976749860e-01 2.20944285602669987511603721941356e-03
            2.65322081006621690590918660745956e-01 6.22451188139388600362167380808387e-01 2.27314527027757664820462046861849e-03
            2.06179798246544199180618761602091e-01 6.72559110565756901323197780584451e-01 2.25048025031910486903896639887535e-03
            1.52756368406658793457353340272675e-01 7.17821771275473419215984449692769e-01 2.13568946995199016036792727391003e-03
            1.06091597010395888300848810104071e-01 7.57358201660716323644351177790668e-01 1.92855214939399071186199030591979e-03
            6.70937111398499097525416345888516e-02 7.90398912110140350328890690434491e-01 1.63458071354281625660320731441288e-03
            3.65216139064131750657793418213259e-02 8.16300926795622050491374466218986e-01 1.26480396535254085120525413543646e-03
            1.49697510822856538759140221372945e-02 8.34560605322337334399662722717039e-01 8.35177066780414196003889948372034e-04
            2.85270725880037989696802469552495e-03 8.44826693535522754707756121206330e-01 3.66166852989810421258043415804195e-04
            9.97147292741199509080729512788821e-01 2.26453665172455338369661426156654e-03 1.17814891063918987023707071054579e-06
            9.85030248917714068568329821573570e-01 1.18832908243392734032317648029675e-02 1.42746950078193613202677536566654e-05
            9.63478386093586713911918195663020e-01 2.89915949195507643976199574353814e-02 5.39205692836340936494737841044866e-05
            9.32906288860150034736307134153321e-01 5.32603433134238232748991492826462e-02 1.32212777831988069444577038602517e-04
            8.93908402989603945165697496122448e-01 8.42176529431389181423739387355454e-02 2.57420197946871471324919999901226e-04
            8.47243631593341484098402816016460e-01 1.21261091187698871740607842184545e-01 4.33064300266350181174007660800385e-04
            7.93820201753455689797078775882255e-01 1.63669689041558596986902784919948e-01 6.57389003529636087135024080652101e-04
            7.34677918993378642475988726801006e-01 2.10618027874322877313062463144888e-01 9.23266046963030963951046814486290e-04
            6.70967910446042203886918287025765e-01 2.61192319713083931365105172517360e-01 1.21854772933154085340357042355208e-03
            6.03930213344110788931118349864846e-01 3.14408197951626111610323732747929e-01 1.52684357352823548738263159663120e-03
            5.34869636659861091843026770220604e-01 3.69229878868327199281651473938837e-01 1.82866394627679228722383353300529e-03
            4.65130363340138908156973229779396e-01 4.24590322885128490515427301943419e-01 2.10284448749866337974312457959059e-03
            3.96069786655889322091184112650808e-01 4.79412003801829467164452580618672e-01 2.32814265609491803654673702794753e-03
            3.29032089553957685090779250458581e-01 5.32627882040371924965427297138376e-01 2.48488354080210211566326350407508e-03
            2.65322081006621690590918660745956e-01 5.83202173879132534928260156448232e-01 2.55652743068573691867517716502789e-03
            2.06179798246544199180618761602091e-01 6.30150512711897148321327222220134e-01 2.53103687097601956837444880932253e-03
            1.52756368406658793457353340272675e-01 6.72559110565756568256290393037489e-01 2.40193567245802243326213698537686e-03
            1.06091597010395888300848810104071e-01 7.09602548810316813288068260590080e-01 2.16897553178887602526314637430005e-03
            6.70937111398499097525416345888516e-02 7.40559858440031915094436953950208e-01 1.83835608154149770382401740675959e-03
            3.65216139064131750657793418213259e-02 7.64828606833904967032822241890244e-01 1.42248103284178394230263808850623e-03
            1.49697510822856538759140221372945e-02 7.81936910929116657520410171855474e-01 9.39294601458997972277809029861828e-04
            2.85270725880037989696802469552495e-03 7.91555665101731253940897659049369e-01 4.11815125111652808719692187011674e-04
            9.97147292741199509080729512788821e-01 2.09582103239285043663753249632009e-03 1.28581157483451368402928814693720e-06
            9.85030248917714068568329821573570e-01 1.09979455729827466092807242148410e-02 1.55791580355733982996176145707778e-05
            9.63478386093586713911918195663020e-01 2.68316233030433115569479696205235e-02 5.88479872794246436731348259829844e-05
            9.32906288860150034736307134153321e-01 4.92922680777678826058441075019800e-02 1.44294764899593639870256644996971e-04
            8.93908402989603945165697496122448e-01 7.79431537142819264030180193003616e-02 2.80944002177706459756623136314602e-04
            8.47243631593341484098402816016460e-01 1.12226730853989764558065189703484e-01 4.72638971951326873795207195172452e-04
            7.93820201753455689797078775882255e-01 1.51475745114245829991972414063639e-01 7.17463117161257473569646592181925e-04
            7.34677918993378642475988726801006e-01 1.94926274336937210840403622569283e-01 1.00763677589168218831638945687246e-03
            6.70967910446042203886918287025765e-01 2.41732610835544758920434560423018e-01 1.32990215473929277063824816451643e-03
            6.03930213344110788931118349864846e-01 2.90983726636500128481799265500740e-01 1.66637096726522399708914079496935e-03
            5.34869636659861091843026770220604e-01 3.41721007399367338042139863318880e-01 1.99577256098261868882959291227053e-03
            4.65130363340138908156973229779396e-01 3.92956911594011304433848863482126e-01 2.29500851521034429647549757191882e-03
            3.96069786655889322091184112650808e-01 4.43694192356878402971886998784612e-01 2.54089508383851039874379296179541e-03
            3.29032089553957685090779250458581e-01 4.92945308157833994577856628893642e-01 2.71195940515334002821057346466205e-03
            2.65322081006621690590918660745956e-01 5.39751644656441209590980179200415e-01 2.79015032146846721911037647601006e-03
            2.06179798246544199180618761602091e-01 5.83202173879132867995167543995194e-01 2.76233036048748829133225868304180e-03
            1.52756368406658793457353340272675e-01 6.22451188139388600362167380808387e-01 2.62143152004346707828474904999894e-03
            1.06091597010395888300848810104071e-01 6.56734765279096799339697554387385e-01 2.36718280611395836404708958866649e-03
            6.70937111398499097525416345888516e-02 6.85385650915610766809038523206254e-01 2.00635039167590444389444037653902e-03
            3.65216139064131750657793418213259e-02 7.07846295690335414185767604067223e-01 1.55247147494979686847826894080526e-03
            1.49697510822856538759140221372945e-02 7.23679973420396138727994639339158e-01 1.02513006618178527731832794245292e-03
            2.85270725880037989696802469552495e-03 7.32582097960985834106395486742258e-01 4.49447985546414397611814095512273e-04
            9.97147292741199509080729512788821e-01 1.91407502855162232890950946284647e-03 1.36844753124817709864812550341062e-06
            9.85030248917714068568329821573570e-01 1.00442226035788069460341631611300e-02 1.65803923140522707917361711071891e-05
            9.63478386093586713911918195663020e-01 2.45048309689032024760724226553066e-02 6.26300030949847330325602756673220e-05
            9.32906288860150034736307134153321e-01 4.50177271675755164004328889859607e-02 1.53568235551385882248220937640326e-04
            8.93908402989603945165697496122448e-01 7.11840571619490186661138864110399e-02 2.98999584172000636931448847022352e-04
            8.47243631593341484098402816016460e-01 1.02494621317141487715218772791559e-01 5.03014319513892752683270614966204e-04
            7.93820201753455689797078775882255e-01 1.38340028405670389943793452403042e-01 7.63572712096097921952164711001387e-04
            7.34677918993378642475988726801006e-01 1.78022602288208264820212889389950e-01 1.07239512021138237371498114214319e-03
            6.70967910446042203886918287025765e-01 2.20769973597714136426262143686472e-01 1.41537170458964109247268492453031e-03
            6.03930213344110788931118349864846e-01 2.65750117143311703760133468676941e-01 1.77346454249427388193127708149177e-03
            5.34869636659861091843026770220604e-01 3.12087547975341383921232818465796e-01 2.12403596877016416152872402278717e-03
            4.65130363340138908156973229779396e-01 3.58880362470700819965685468559968e-01 2.44250308388874225531961492663413e-03
            3.96069786655889322091184112650808e-01 4.05217793302730389104482355833170e-01 2.70419217923663964681102456211192e-03
            3.29032089553957685090779250458581e-01 4.50197936848328206238534221483860e-01 2.88625038494073156078445485661632e-03
            2.65322081006621690590918660745956e-01 4.92945308157833717022100472604507e-01 2.96946643968128663748640150288338e-03
            2.06179798246544199180618761602091e-01 5.32627882040371924965427297138376e-01 2.93985855803755458809378708906479e-03
            1.52756368406658793457353340272675e-01 5.68473289128900494127094589202898e-01 2.78990449467786871470820209140129e-03
            1.06091597010395888300848810104071e-01 5.99783853284093226854167824058095e-01 2.51931583945858581299881429060861e-03
            6.70937111398499097525416345888516e-02 6.25950183278466742997636629297631e-01 2.13529361069958196345242029678957e-03
            3.65216139064131750657793418213259e-02 6.46463079477138991002505008509615e-01 1.65224500915049665218659047383198e-03
            1.49697510822856538759140221372945e-02 6.60923687842463580821572577406187e-01 1.09101266136548205598622018897004e-03
            2.85270725880037989696802469552495e-03 6.69053835417490638803883484797552e-01 4.78332905289497595235914673139632e-04
            9.97147292741199509080729512788821e-01 1.72283610341567396985473781256815e-03 1.42444830324424032089050771304750e-06
            9.85030248917714068568329821573570e-01 9.04068496483321007439748484557640e-03 1.72589092088415157891841816839573e-05
            9.63478386093586713911918195663020e-01 2.20565060781713860593722387193338e-02 6.51930012687151514642197591697936e-05
            9.32906288860150034736307134153321e-01 4.05199192827377652204035030081286e-02 1.59852685300881783899556998207458e-04
            8.93908402989603945165697496122448e-01 6.40719208165059089798987201902492e-02 3.11235499073948926503085932182557e-04
            8.47243631593341484098402816016460e-01 9.22541861615048669476024656432855e-02 5.23599098670283883728837626136965e-04
            7.93820201753455689797078775882255e-01 1.24518209542281224622151114544977e-01 7.94820283066909440486402793624165e-04
            7.34677918993378642475988726801006e-01 1.60236020987232274936928888564580e-01 1.11628058402735456586252205113396e-03
            6.70967910446042203886918287025765e-01 1.98712420041380344315484762773849e-01 1.47329274745644956019774429734071e-03
            6.03930213344110788931118349864846e-01 2.39198510754247611886924573809665e-01 1.84603976457585255629978426838989e-03
            5.34869636659861091843026770220604e-01 2.80906279564833882833596589989611e-01 2.21095757247245696536230852302651e-03
            4.65130363340138908156973229779396e-01 3.23023933779276906097521759875235e-01 2.54245726932672859096906670117733e-03
            3.96069786655889322091184112650808e-01 3.64731702589863093777466929168440e-01 2.81485542806784701588829200602504e-03
            3.29032089553957685090779250458581e-01 4.05217793302730555637936049606651e-01 3.00436397427447004007672326508782e-03
            2.65322081006621690590918660745956e-01 4.43694192356878291949584536268958e-01 3.09098546707642233069557313740461e-03
            2.06179798246544199180618761602091e-01 4.79412003801829633697906274392153e-01 3.06016594655625882859251163381487e-03
            1.52756368406658793457353340272675e-01 5.11676027182605741572274382633623e-01 2.90407533567075800198797885798285e-03
            1.06091597010395888300848810104071e-01 5.39858292527604977095734284375794e-01 2.62241342171145187106562346457395e-03
            6.70937111398499097525416345888516e-02 5.63410294061373106977441693743458e-01 2.22267590918520186252238524104996e-03
            3.65216139064131750657793418213259e-02 5.81873707265939454913450390449725e-01 1.71985958254570418107798346341042e-03
            1.49697510822856538759140221372945e-02 5.94889528379277776615197126375278e-01 1.13566000801107740321438388519937e-03
            2.85270725880037989696802469552495e-03 6.02207377240695174158702229760820e-01 4.97907650652879463400168891240583e-04
            9.97147292741199509080729512788821e-01 1.52582649501156698487736029790085e-03 1.45272387087859875227458059804286e-06
            9.85030248917714068568329821573570e-01 8.00686532227086814050842633605498e-03 1.76015018136544791344921118136924e-05
            9.63478386093586713911918195663020e-01 1.95343023603549749422914061369738e-02 6.64870946468054777218903428348540e-05
            9.32906288860150034736307134153321e-01 3.58863888995332586118180984158244e-02 1.63025791270726097981907942724433e-04
            8.93908402989603945165697496122448e-01 5.67451739456149434337461912036815e-02 3.17413582465417482871006260580771e-04
            8.47243631593341484098402816016460e-01 8.17047432671493267797302451072028e-02 5.33992639589966010781052485612008e-04
            7.93820201753455689797078775882255e-01 1.10279313774732615205387276091642e-01 8.10597615680408271898738092886560e-04
            7.34677918993378642475988726801006e-01 1.41912725065849787409177906738478e-01 1.13843896427934373680179724885875e-03
            6.70967910446042203886918287025765e-01 1.75989274189160299632916917289549e-01 1.50253788652610640111384476824696e-03
            6.03930213344110788931118349864846e-01 2.11845702880584157989929394716455e-01 1.88268400227833499274709971871289e-03
            5.34869636659861091843026770220604e-01 2.48784108439209283858417620649561e-01 2.25484549752720227558389076705225e-03
            4.65130363340138908156973229779396e-01 2.86085528220651807984609149571043e-01 2.59292552592304235969944414819111e-03
            3.96069786655889322091184112650808e-01 3.23023933779276850586370528617408e-01 2.87073083952082887954082579540227e-03
            3.29032089553957685090779250458581e-01 3.58880362470700875476836699817795e-01 3.06400116613278483690407938411226e-03
            2.65322081006621690590918660745956e-01 3.92956911594011137900395169708645e-01 3.15234211191364310095108791642815e-03
            2.06179798246544199180618761602091e-01 4.24590322885128546026578533201246e-01 3.12091081809504414246170256319601e-03
            1.52756368406658793457353340272675e-01 4.53164893392711598529842831339920e-01 2.96172177912680479813789524712320e-03
            1.06091597010395888300848810104071e-01 4.78124462714246190042644002460293e-01 2.67446882302153909893860372903873e-03
            6.70937111398499097525416345888516e-02 4.98983247760327874864572095248150e-01 2.26679644543512255616679595959795e-03
            3.65216139064131750657793418213259e-02 5.15335334299506175881333547295071e-01 1.75399911982277757760417813415188e-03
            1.49697510822856538759140221372945e-02 5.26862771337590407583206797426101e-01 1.15820307348633399872894766957643e-03
            2.85270725880037989696802469552495e-03 5.33343810164849552180044156557415e-01 5.07791211481050937075198969949952e-04
            9.97147292741199509080729512788821e-01 1.32688076378892393439312691327814e-03 1.45272387087860743436228995462134e-06
            9.85030248917714068568329821573570e-01 6.96288576001511880231298334820167e-03 1.76015018136545841665775713469344e-05
            9.63478386093586713911918195663020e-01 1.69873115460582556346391669421791e-02 6.64870946468058707451778688302113e-05
            9.32906288860150034736307134153321e-01 3.12073222403167656324729506422955e-02 1.63025791270727073763863179678424e-04
            8.93908402989603945165697496122448e-01 4.93464230647811114005563126738707e-02 3.17413582465419380224808110213530e-04
            8.47243631593341484098402816016460e-01 7.10516251395091891218669388763374e-02 5.33992639589969154967352693574867e-04
            7.93820201753455689797078775882255e-01 9.59004844718116949975339480261027e-02 8.10597615680413042388297029106070e-04
            7.34677918993378642475988726801006e-01 1.23409355940771556237045558646059e-01 1.13843896427935045885526665898624e-03
            6.70967910446042203886918287025765e-01 1.53042815364797551991316026942513e-01 1.50253788652611529157165914938332e-03
            6.03930213344110788931118349864846e-01 1.84224083775305053078952255418699e-01 1.88268400227834626844969356795900e-03
            5.34869636659861091843026770220604e-01 2.16346254900929624298555609129835e-01 2.25484549752721571969082958730723e-03
            4.65130363340138908156973229779396e-01 2.48784108439209283858417620649561e-01 2.59292552592305797221072793945496e-03
            3.96069786655889322091184112650808e-01 2.80906279564833827322445358731784e-01 2.87073083952084579309471656927144e-03
            3.29032089553957685090779250458581e-01 3.12087547975341439432384049723623e-01 3.06400116613280305150057714058676e-03
            2.65322081006621690590918660745956e-01 3.41721007399367171508686169545399e-01 3.15234211191366218290932366130619e-03
            2.06179798246544199180618761602091e-01 3.69229878868327254792802705196664e-01 3.12091081809506279073906931387228e-03
            1.52756368406658793457353340272675e-01 3.94078738200629552501652597129578e-01 2.96172177912682257905352400939591e-03
            1.06091597010395888300848810104071e-01 4.15783940275357866145355956177809e-01 2.67446882302155514513075651450436e-03
            6.70937111398499097525416345888516e-02 4.33923041099822159871735038905172e-01 2.26679644543513600027373477985293e-03
            3.65216139064131750657793418213259e-02 4.48143051794080704564038342141430e-01 1.75399911982278798594503399499445e-03
            1.49697510822856538759140221372945e-02 4.58167477580123938540879180436605e-01 1.15820307348634093762285157680481e-03
            2.85270725880037989696802469552495e-03 4.63803482576350012411836587489233e-01 5.07791211481053972841281929362367e-04
            9.97147292741199509080729512788821e-01 1.12987115538481738309661839281262e-03 1.42444830324423714451695550942123e-06
            9.85030248917714068568329821573570e-01 5.92906611745277947050913880389089e-03 1.72589092088414785197345024947424e-05
            9.63478386093586713911918195663020e-01 1.44651078282418479870052863134333e-02 6.51930012687150023864210424129340e-05
            9.32906288860150034736307134153321e-01 2.65737918571122624933344980036054e-02 1.59852685300881404428796628280907e-04
            8.93908402989603945165697496122448e-01 4.20196761938901527932976875945315e-02 3.11235499073948221771673816604675e-04
            8.47243631593341484098402816016460e-01 6.05021822451536697706764300619398e-02 5.23599098670282691106447892082087e-04
            7.93820201753455689797078775882255e-01 8.16615887042631133363457252016815e-02 7.94820283066907705762926816817071e-04
            7.34677918993378642475988726801006e-01 1.05086060019389110342658000263327e-01 1.11628058402735196377730808592332e-03
            6.70967910446042203886918287025765e-01 1.30319669512577562819899412716040e-01 1.47329274745644630759122684082740e-03
            6.03930213344110788931118349864846e-01 1.56871275901641626937532691954402e-01 1.84603976457584843633152882347304e-03
            5.34869636659861091843026770220604e-01 1.84224083775305108590103486676526e-01 2.21095757247245219487274958680700e-03
            4.65130363340138908156973229779396e-01 2.11845702880584213501080625974282e-01 2.54245726932672295311776977655427e-03
            3.96069786655889322091184112650808e-01 2.39198510754247639642500189438579e-01 2.81485542806784094435612608720021e-03
            3.29032089553957685090779250458581e-01 2.65750117143311814782435931192595e-01 3.00436397427446353486368835206122e-03
            2.65322081006621690590918660745956e-01 2.90983726636500072970648034242913e-01 3.09098546707641539180166923017623e-03
            2.06179798246544199180618761602091e-01 3.14408197951626278143777426521410e-01 3.06016594655625232337947672078826e-03
            1.52756368406658793457353340272675e-01 3.35567604410735520481523508351529e-01 2.90407533567075149677494394495625e-03
            1.06091597010395888300848810104071e-01 3.54050110461999190114568136777962e-01 2.62241342171144623321432653995089e-03
            6.70937111398499097525416345888516e-02 3.69495994798777094292319134183344e-01 2.22267590918519709203282630483045e-03
            3.65216139064131750657793418213259e-02 3.81604678827647481043072730244603e-01 1.71985958254570049479059701269534e-03
            1.49697510822856538759140221372945e-02 3.90140720538436680531191314003081e-01 1.13566000801107480112916991998873e-03
            2.85270725880037989696802469552495e-03 3.94939915500504556966632208059309e-01 4.97907650652878379197996405736149e-04
            9.97147292741199509080729512788821e-01 9.38632230248868265100326002681186e-04 1.36844753124817646337341506268537e-06
            9.85030248917714068568329821573570e-01 4.92552847870717826206377054631957e-03 1.65803923140522640154725930727864e-05
            9.63478386093586713911918195663020e-01 1.20167829375100246314111984702322e-02 6.26300030949847059275059635297112e-05
            9.32906288860150034736307134153321e-01 2.20759839722745009049642561649307e-02 1.53568235551385800933058001227494e-04
            8.93908402989603945165697496122448e-01 3.49075398484470222904008096520556e-02 2.98999584172000528511231598471909e-04
            8.47243631593341484098402816016460e-01 5.02617470895170143085906033775245e-02 5.03014319513892535842836117865318e-04
            7.93820201753455689797078775882255e-01 6.78397698408738925035521560857887e-02 7.63572712096097596691512965350057e-04
            7.34677918993378642475988726801006e-01 8.72994787184130510704349603656738e-02 1.07239512021138194003411214794141e-03
            6.70967910446042203886918287025765e-01 1.08262115956243687442395184916677e-01 1.41537170458964044195138143322765e-03
            6.03930213344110788931118349864846e-01 1.30319669512577451797596950200386e-01 1.77346454249427323140997359018911e-03
            5.34869636659861091843026770220604e-01 1.53042815364797468724589180055773e-01 2.12403596877016329416698603438363e-03
            4.65130363340138908156973229779396e-01 1.75989274189160216366190070402808e-01 2.44250308388874138795787693823058e-03
            3.96069786655889322091184112650808e-01 1.98712420041380205537606684629282e-01 2.70419217923663877944928657370838e-03
            3.29032089553957685090779250458581e-01 2.20769973597714053159535296799731e-01 2.88625038494073025974184787401100e-03
            2.65322081006621690590918660745956e-01 2.41732610835544509120254019762797e-01 2.96946643968128533644379452027806e-03
            2.06179798246544199180618761602091e-01 2.61192319713083820342802710001706e-01 2.93985855803755328705118010645947e-03
            1.52756368406658793457353340272675e-01 2.78770342464440545882098376750946e-01 2.78990449467786784734646410299774e-03
            1.06091597010395888300848810104071e-01 2.94124549705510718311529672064353e-01 2.51931583945858451195620730800329e-03
            6.70937111398499097525416345888516e-02 3.06956105581683180716368042340036e-01 2.13529361069958109609068230838602e-03
            3.65216139064131750657793418213259e-02 3.17015306616447667398261955895578e-01 1.65224500915049600166528698252932e-03
            1.49697510822856538759140221372945e-02 3.24106561075250598769059706683038e-01 1.09101266136548162230535119476826e-03
            2.85270725880037989696802469552495e-03 3.28093457323708870276846027991269e-01 4.78332905289497378395480176038745e-04
            9.97147292741199509080729512788821e-01 7.56886226407641566835127200363331e-04 1.28581157483452956589704916506856e-06
            9.85030248917714068568329821573570e-01 3.97180550930324553771111339983690e-03 1.55791580355735914231295885512552e-05
            9.63478386093586713911918195663020e-01 9.68999060336993289777041127308621e-03 5.88479872794253755096012536984773e-05
            9.32906288860150034736307134153321e-01 1.78014430620821624551286532778249e-02 1.44294764899595401698786933941676e-04
            8.93908402989603945165697496122448e-01 2.81484432961141665952009560669467e-02 2.80944002177709929203575089928790e-04
            8.47243631593341484098402816016460e-01 4.05296375526687999157893216306547e-02 4.72638971951332728486938616896396e-04
            7.93820201753455689797078775882255e-01 5.47040531322985495998878491263895e-02 7.17463117161266364027460973318284e-04
            7.34677918993378642475988726801006e-01 7.03958066696842299503344975164509e-02 1.00763677589169454822115579162300e-03
            6.70967910446042203886918287025765e-01 8.72994787184132037261008463246981e-02 1.32990215473930925051126994418382e-03
            6.03930213344110788931118349864846e-01 1.05086060019389207487172654964525e-01 1.66637096726524459693041801955360e-03
            5.34869636659861091843026770220604e-01 1.23409355940771708892711444605084e-01 1.99577256098264340863912558177162e-03
            4.65130363340138908156973229779396e-01 1.41912725065849981698207216140872e-01 2.29500851521037291941285118923588e-03
            3.96069786655889322091184112650808e-01 1.60236020987232441470382582338061e-01 2.54089508383854162376636054432311e-03
            3.29032089553957685090779250458581e-01 1.78022602288208514620393430050171e-01 2.71195940515337385531835501240039e-03
            2.65322081006621690590918660745956e-01 1.94926274336937377373857316342765e-01 2.79015032146850147989902701795017e-03
            2.06179798246544199180618761602091e-01 2.10618027874323182624394235062937e-01 2.76233036048752255212090922498192e-03
            1.52756368406658793457353340272675e-01 2.24792443453952800469508588321332e-01 2.62143152004349960434992361513196e-03
            1.06091597010395888300848810104071e-01 2.37173637710507562159634176168765e-01 2.36718280611398785434618119438710e-03
            6.70937111398499097525416345888516e-02 2.47520637944539545483024767236202e-01 2.00635039167592916370397304604012e-03
            3.65216139064131750657793418213259e-02 2.55632090403251799326511672916240e-01 1.55247147494981595043650468568330e-03
            1.49697510822856538759140221372945e-02 2.61350275497318540462998726070509e-01 1.02513006618179785406352877430436e-03
            2.85270725880037989696802469552495e-03 2.64565194780214063552392644851352e-01 4.49447985546419981253002395860108e-04
            9.97147292741199509080729512788821e-01 5.88170607075937318733438452511564e-04 1.17814891063919347012709654132223e-06
            9.85030248917714068568329821573570e-01 3.08646025794671137118529991028026e-03 1.42746950078194036719151163716823e-05
            9.63478386093586713911918195663020e-01 7.53001898686246270986366369015741e-03 5.39205692836342495035360788957490e-05
            9.32906288860150034736307134153321e-01 1.38333678264261922957745198914381e-02 1.32212777831988476020391720666680e-04
            8.93908402989603945165697496122448e-01 2.18739440672571297530346612347785e-02 2.57420197946872230266440739754330e-04
            8.47243631593341484098402816016460e-01 3.14952772189596302832015339845384e-02 4.33064300266351482216614643405705e-04
            7.93820201753455689797078775882255e-01 4.25101092049856923993367274761113e-02 6.57389003529638038698934554560083e-04
            7.34677918993378642475988726801006e-01 5.47040531322984455164792905179638e-02 9.23266046963033674456478028247375e-04
            6.70967910446042203886918287025765e-01 6.78397698408738925035521560857887e-02 1.21854772933154453969095687426716e-03
            6.03930213344110788931118349864846e-01 8.16615887042630439474066861293977e-02 1.52684357352823982419132153864894e-03
            5.34869636659861091843026770220604e-01 9.59004844718116394863827167682757e-02 1.82866394627679770823469596052746e-03
            4.65130363340138908156973229779396e-01 1.10279313774732559694236044833815e-01 2.10284448749866945127529049841542e-03
            3.96069786655889322091184112650808e-01 1.24518209542281141355424267658236e-01 2.32814265609492497544064093517591e-03
            3.29032089553957685090779250458581e-01 1.38340028405670334432642221145215e-01 2.48488354080210948823803640550523e-03
            2.65322081006621690590918660745956e-01 1.51475745114245691214094335919071e-01 2.55652743068574429124995006645804e-03
            2.06179798246544199180618761602091e-01 1.63669689041558541475751553662121e-01 2.53103687097602694094922171075268e-03
            1.52756368406658793457353340272675e-01 1.74684521027584499508478188545268e-01 2.40193567245802980583690988680701e-03
            1.06091597010395888300848810104071e-01 1.84305854179287104122053619903454e-01 2.16897553178888209679531229312488e-03
            6.70937111398499097525416345888516e-02 1.92346430420118008619567717687460e-01 1.83835608154150312483487983428176e-03
            3.65216139064131750657793418213259e-02 1.98649779259681746879095953772776e-01 1.42248103284178806227089353342308e-03
            1.49697510822856538759140221372945e-02 2.03093337988597549825797727862664e-01 9.39294601459000682783240243622913e-04
            2.85270725880037989696802469552495e-03 2.05591627639468282895407469368365e-01 4.11815125111654001342081921066551e-04
            9.97147292741199509080729512788821e-01 4.35769200981677502955424774455651e-04 1.04755520780147249644423737557908e-06
            9.85030248917714068568329821573570e-01 2.28672481128165726652068379110005e-03 1.26923947899811761412872113119477e-05
            9.63478386093586713911918195663020e-01 5.57890910869381258241350707294259e-03 4.79436620028328127143812942279055e-05
            9.32906288860150034736307134153321e-01 1.02489916566488765969911156616945e-02 1.17557451952874647007306341262023e-04
            8.93908402989603945165697496122448e-01 1.62061670777708388690996343939332e-02 2.28886065689463204285772346402439e-04
            8.47243631593341484098402816016460e-01 2.33345080887908222500026766965675e-02 3.85060631096954320874703459054444e-04
            7.93820201753455689797078775882255e-01 3.14952772189597066110344769640506e-02 5.84519722405270184852066339686871e-04
            7.34677918993378642475988726801006e-01 4.05296375526688207324710333523399e-02 8.20925221717240072120957972146016e-04
            6.70967910446042203886918287025765e-01 5.02617470895171322697869698004070e-02 1.08347595816505725546674554493620e-03
            6.03930213344110788931118349864846e-01 6.05021822451537599762971808559087e-02 1.35759828193530323638127388363728e-03
            5.34869636659861091843026770220604e-01 7.10516251395093140219572092064482e-02 1.62596291771109516081927903741189e-03
            4.65130363340138908156973229779396e-01 8.17047432671494794353961310662271e-02 1.86975150100568920062682476412874e-03
            3.96069786655889322091184112650808e-01 9.22541861615050196032683516023098e-02 2.07007615240573615567054588382234e-03
            3.29032089553957685090779250458581e-01 1.02494621317141682004248082193953e-01 2.20944285602670638032907213244016e-03
            2.65322081006621690590918660745956e-01 1.12226730853989917213731075662508e-01 2.27314527027758315341765538164509e-03
            2.06179798246544199180618761602091e-01 1.21261091187699107663000575030310e-01 2.25048025031911137425200131190195e-03
            1.52756368406658793457353340272675e-01 1.29421860317867926104540288179123e-01 2.13568946995199623190009319273486e-03
            1.06091597010395888300848810104071e-01 1.36550201328887982343829321507656e-01 1.92855214939399634971328723054285e-03
            6.70937111398499097525416345888516e-02 1.42507376750009934207596984379052e-01 1.63458071354282102709276625063239e-03
            3.65216139064131750657793418213259e-02 1.47177459297964996487451116990997e-01 1.26480396535254453749264058615154e-03
            1.49697510822856538759140221372945e-02 1.50469643595377178257876948919147e-01 8.35177066780416581248669416481789e-04
            2.85270725880037989696802469552495e-03 1.52320599205677142951032010387280e-01 3.66166852989811451250107277033408e-04
            9.97147292741199509080729512788821e-01 3.02648268889292809766949909899836e-04 8.96572665723638424833876348768458e-07
            9.85030248917714068568329821573570e-01 1.58816479916782261200269843470778e-03 1.08630591939426052715007148252724e-05
            9.63478386093586713911918195663020e-01 3.87463634472846266826318029075082e-03 4.10336147692363231785465937928592e-05
            9.32906288860150034736307134153321e-01 7.11807896418087740486591030730779e-03 1.00614074836466311211637114642059e-04
            8.93908402989603945165697496122448e-01 1.12554269562162596179932805284807e-02 1.95897064454371742578397963541192e-04
            8.47243631593341484098402816016460e-01 1.62061670777707868273953550897204e-02 3.29562426797891556973230109406359e-04
            7.93820201753455689797078775882255e-01 2.18739440672571089363529495130933e-02 5.00273782023192810711198497131136e-04
            7.34677918993378642475988726801006e-01 2.81484432961140867979210611338203e-02 7.02606515545366422208217738187841e-04
            6.70967910446042203886918287025765e-01 3.49075398484469875959312901159137e-02 9.27316212859319332663388113502378e-04
            6.03930213344110788931118349864846e-01 4.20196761938900764654647446150193e-02 1.16192970217875277301466940826913e-03
            5.34869636659861091843026770220604e-01 4.93464230647810350727233696943586e-02 1.39161535033511629504132667278782e-03
            4.65130363340138908156973229779396e-01 5.67451739456148532281254404097126e-02 1.60026705515184841828457695811494e-03
            3.96069786655889322091184112650808e-01 6.40719208165057979575962576745951e-02 1.77171921860664249512296031241476e-03
            3.29032089553957685090779250458581e-01 7.11840571619489215215992317098426e-02 1.89099921077126521991307317449582e-03
            2.65322081006621690590918660745956e-01 7.79431537142817737473521333413373e-02 1.94552029274632030247937919398282e-03
            2.06179798246544199180618761602091e-01 8.42176529431387932422836684054346e-02 1.92612194771255929373610094046398e-03
            1.52756368406658793457353340272675e-01 8.98854299326250633095369835245947e-02 1.82787578828556515099512669308979e-03
            1.06091597010395888300848810104071e-01 9.48361700541796442953668133668543e-02 1.65059285533800332093634644081703e-03
            6.70937111398499097525416345888516e-02 9.89735180462150143654298517503776e-02 1.39899107633406193790248650543617e-03
            3.65216139064131750657793418213259e-02 1.02216960665667433438841271708952e-01 1.08250968959801795697683868269223e-03
            1.49697510822856538759140221372945e-02 1.04503432211228097781230417240295e-01 7.14804263811626822665290514891012e-04
            2.85270725880037989696802469552495e-03 1.05788948741506613759888466574921e-01 3.13391780251573903639389362396628e-04
            9.97147292741199509080729512788821e-01 1.91398716788513187240031165892162e-04 7.28140984419260427489180298454841e-07
            9.85030248917714068568329821573570e-01 1.00437615495035156149006283499148e-03 8.82230623091537934825703826113852e-06
            9.63478386093586713911918195663020e-01 2.45037061379801491059482465573183e-03 3.33249694024936770806420704982287e-05
            9.32906288860150034736307134153321e-01 4.50156607451762755950364081058979e-03 8.17125418816191316203137451346095e-05
            8.93908402989603945165697496122448e-01 7.11807896418087740486591030730779e-03 1.59095505372698300299363904919403e-04
            8.47243631593341484098402816016460e-01 1.02489916566488419025215961255526e-02 2.67650263107826514880766000104018e-04
            7.93820201753455689797078775882255e-01 1.38333678264261766832632361001743e-02 4.06291489856548313202438960090035e-04
            7.34677918993378642475988726801006e-01 1.78014430620821104134243739736121e-02 5.70613648449422175903955611886431e-04
            6.70967910446042203886918287025765e-01 2.20759839722744766188355924896314e-02 7.53108995972285818511215982340445e-04
            6.03930213344110788931118349864846e-01 2.65737918571122104516302186993926e-02 9.43647592119658900310097404684484e-04
            5.34869636659861091843026770220604e-01 3.12073222403167135907686713380826e-02 1.13018409981094055186123803480314e-03
            4.65130363340138908156973229779396e-01 3.58863888995331961617729632507690e-02 1.29963813689491607117343363597683e-03
            3.96069786655889322091184112650808e-01 4.05199192827376888925705600286165e-02 1.43888100236642165613565325088530e-03
            3.29032089553957685090779250458581e-01 4.50177271675754470114938499136770e-02 1.53575285027868191876920800353901e-03
            2.65322081006621690590918660745956e-01 4.92922680777677785224355488935544e-02 1.58003150812609477297854265742671e-03
            2.06179798246544199180618761602091e-01 5.32603433134237400081723023959057e-02 1.56427736951694263713152910355575e-03
            1.52756368406658793457353340272675e-01 5.68447194832010435638913747879997e-02 1.48448790238786803435955263807955e-03
            1.06091597010395888300848810104071e-01 5.99756321756690427560165801423864e-02 1.34050964579783093814924210818162e-03
            6.70937111398499097525416345888516e-02 6.25921450653322830603997317666654e-02 1.13617420925214728834318211880827e-03
            3.65216139064131750657793418213259e-02 6.46433405260519039492450588113570e-02 8.79147559546781349679167583133221e-04
            1.49697510822856538759140221372945e-02 6.60893349848995781403715454871417e-02 5.80519906770519500231741982787526e-04
            2.85270725880037989696802469552495e-03 6.69023124230614107954551172952051e-02 2.54517462003045740609463720716121e-04
            9.97147292741199509080729512788821e-01 1.04185473093933812565800578475717e-04 5.45542222925193825938227753374887e-07
            9.85030248917714068568329821573570e-01 5.46719469302359541498792516733829e-04 6.60990200459461298512255089665501e-06
            9.63478386093586713911918195663020e-01 1.33382828232911430747786862838211e-03 2.49679365339534326928127433431470e-05
            9.32906288860150034736307134153321e-01 2.45037061379801491059482465573183e-03 6.12211682803709498351674156424451e-05
            8.93908402989603945165697496122448e-01 3.87463634472846310194404928495260e-03 1.19198503470659917320879916946552e-04
            8.47243631593341484098402816016460e-01 5.57890910869379350045527132806455e-03 2.00530560189264339720008334388979e-04
            7.93820201753455689797078775882255e-01 7.53001898686245577096975978292903e-03 3.04404184457093514911546483503457e-04
            7.34677918993378642475988726801006e-01 9.68999060336990514219479564417270e-03 4.27518632885127113085754757548784e-04
            6.70967910446042203886918287025765e-01 1.20167829375100124883468666325825e-02 5.64248908602999962869084527028463e-04
            6.03930213344110788931118349864846e-01 1.44651078282418219661531466613269e-02 7.07005396040921003063262073595752e-04
            5.34869636659861091843026770220604e-01 1.69873115460582313485105032668798e-02 8.46763414391951360106480795053585e-04
            4.65130363340138908156973229779396e-01 1.95343023603549437172688385544461e-02 9.73722800077637820961862225033201e-04
            3.96069786655889322091184112650808e-01 2.20565060781713478954557672295778e-02 1.07804718777349613396954186583798e-03
            3.29032089553957685090779250458581e-01 2.45048309689031677816029031191647e-02 1.15062610364248161208544640743412e-03
            2.65322081006621690590918660745956e-01 2.68316233030432560457967383626965e-02 1.18380082934410862244289930345076e-03
            2.06179798246544199180618761602091e-01 2.89915949195507227642565339920111e-02 1.17199741766832823443300615195994e-03
            1.52756368406658793457353340272675e-01 3.09427047977193676875362626788046e-02 1.11221706716611227865054711827497e-03
            1.06091597010395888300848810104071e-01 3.26469775616847132648778995189787e-02 1.00434480089659503368459958494441e-03
            6.70937111398499097525416345888516e-02 3.40712432926151623235888621366030e-02 8.51251360668904056595851503885797e-04
            3.65216139064131750657793418213259e-02 3.51877856240840614088227766842465e-02 6.58680288813754950642820329420601e-04
            1.49697510822856538759140221372945e-02 3.59748944371108270168235776509391e-02 4.34940660076293787467821561065762e-04
            2.85270725880037989696802469552495e-03 3.64174284333192432860926146531710e-02 1.90691122963174494600938069233109e-04
            9.97147292741199509080729512788821e-01 4.27043175748727878203794827705053e-05 3.52351496340037805576916353500483e-07
            9.85030248917714068568329821573570e-01 2.24093447465597504211429202847228e-04 4.26916334631588606328541843359936e-06
            9.63478386093586713911918195663020e-01 5.46719469302348265796198667487715e-04 1.61261391484778330836311749374445e-05
            9.32906288860150034736307134153321e-01 1.00437615495033096164878561040723e-03 3.95411562016381037758770700119015e-05
            8.93908402989603945165697496122448e-01 1.58816479916778986909708937247387e-03 7.69872051959199750828521935375193e-05
            8.47243631593341484098402816016460e-01 2.28672481128160218905032152747481e-03 1.29517459832402407129969823529336e-04
            7.93820201753455689797078775882255e-01 3.08646025794664458433147480320713e-03 1.96606725159627435811637852225431e-04
            7.34677918993378642475988726801006e-01 3.97180550930315186264341065225381e-03 2.76123137092135900738654186170606e-04
            6.70967910446042203886918287025765e-01 4.92552847870707070920825998427972e-03 3.64433656827625870530673601876970e-04
            6.03930213344110788931118349864846e-01 5.92906611745264589680148858974462e-03 4.56636349574831711291278413256123e-04
            5.34869636659861091843026770220604e-01 6.96288576001496354456188342396672e-03 5.46902409326280921492147157891850e-04
            4.65130363340138908156973229779396e-01 8.00686532227068946399040072492426e-03 6.28902165973710446447708655171027e-04
            3.96069786655889322091184112650808e-01 9.04068496483300711175079555914635e-03 6.96282567645084581910153165296151e-04
            3.29032089553957685090779250458581e-01 1.00442226035785849014292381298219e-02 7.43159396851907311058738070386198e-04
            2.65322081006621690590918660745956e-01 1.09979455729824968091001835546194e-02 7.64586087125231008039538682652392e-04
            2.06179798246544199180618761602091e-01 1.18832908243390114599868923050963e-02 7.56962571307191463103325457240089e-04
            1.52756368406658793457353340272675e-01 1.26830262710040473500550106678020e-02 7.18352001737994089874961289865496e-04
            1.06091597010395888300848810104071e-01 1.33815862831178650910191052503251e-02 6.48680117809649191967791281854261e-04
            6.70937111398499097525416345888516e-02 1.39653749273353239984674090123917e-02 5.49801056799796247255562864353351e-04
            3.65216139064131750657793418213259e-02 1.44230316129833058269582579669077e-02 4.25424422932410160793498254960809e-04
            1.49697510822856538759140221372945e-02 1.47456576348200599224647078244743e-02 2.80916830919648494689633277943130e-04
            2.85270725880037989696802469552495e-03 1.49270467647107819147977991747211e-02 1.23162423899222185684834696139944e-04
            9.97147292741199509080729512788821e-01 8.13793870441269504088354497239877e-06 1.52604324215665168245308199942989e-07
            9.85030248917714068568329821573570e-01 4.27043175748720763127037891582205e-05 1.84898544265609851065357109589637e-06
            9.63478386093586713911918195663020e-01 1.04185473093929922990506786728560e-04 6.98427165067670272945079407156399e-06
            9.32906288860150034736307134153321e-01 1.91398716788506058610747073700509e-04 1.71253747565577693786047913659942e-05
            8.93908402989603945165697496122448e-01 3.02648268889281534064356060653722e-04 3.33433532827631670474553826633013e-05
            8.47243631593341484098402816016460e-01 4.35769200981659776249904636458155e-04 5.60943394228674728691809436842419e-05
            7.93820201753455689797078775882255e-01 5.88170607075914767328250754019336e-04 8.51508699150959274355374151177500e-05
            7.34677918993378642475988726801006e-01 7.56886226407611100754080357688736e-04 1.19589629032226015935974627879546e-04
            6.70967910446042203886918287025765e-01 9.38632230248832269588199483933977e-04 1.57837138480470336350525162316671e-04
            6.03930213344110788931118349864846e-01 1.12987115538477293080754648713082e-03 1.97770357903989240375489111478657e-04
            5.34869636659861091843026770220604e-01 1.32688076378887232636971660326708e-03 2.36864816679005144159017581806381e-04
            4.65130363340138908156973229779396e-01 1.52582649501150757059830809225787e-03 2.72379118672925926280398156720253e-04
            3.96069786655889322091184112650808e-01 1.72283610341560653247960921419235e-03 3.01561741052133660146578053584676e-04
            3.29032089553957685090779250458581e-01 1.91407502855154816948091145434319e-03 3.21864214340275285423931261874486e-04
            2.65322081006621690590918660745956e-01 2.09582103239276760359155460378133e-03 3.31144168089026759947690603524961e-04
            2.06179798246544199180618761602091e-01 2.26453665172446534648020843860650e-03 3.27842404107196933645690384295790e-04
            1.52756368406658793457353340272675e-01 2.41693805781871898230583894928714e-03 3.11120068774745138322213033887920e-04
            1.06091597010395888300848810104071e-01 2.55005898991109885079464625334822e-03 2.80944999634534476554842408191348e-04
            6.70937111398499097525416345888516e-02 2.66130854201187400098760349465010e-03 2.38120228230907131708460333641142e-04
            3.65216139064131750657793418213259e-02 2.74852178570645028568564249837891e-03 1.84252393535422928205064208029285e-04
            1.49697510822856538759140221372945e-02 2.81000294122550880321353972135512e-03 1.21665789952906197189316794826652e-04
            2.85270725880037989696802469552495e-03 2.84456932009596744417856406528244e-03 5.33419572873498093229689687166939e-05
            ];
        
    case  42
        
        % ALG. DEG.:   42
        % PTS CARD.:  484
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.9e-15
        
        xyw=[
            9.97147292741199509080729512788821e-01 2.84456932009607803280015758673471e-03 1.52604324215667471116133547572036e-07
            9.85030248917714068568329821573570e-01 1.49270467647111132469817107448762e-02 1.84898544265612625098259367423248e-06
            9.63478386093586713911918195663020e-01 3.64174284333192987972438459109981e-02 6.98427165067680691450330635050570e-06
            9.32906288860150034736307134153321e-01 6.69023124230615079399697719964024e-02 1.71253747565580234884889676560960e-05
            8.93908402989603945165697496122448e-01 1.05788948741506766415554352533945e-01 3.33433532827636684909601572091020e-05
            8.47243631593341484098402816016460e-01 1.52320599205676837639700238469231e-01 5.60943394228683063496010419157756e-05
            7.93820201753455689797078775882255e-01 2.05591627639468366162134316255106e-01 8.51508699150972013730900855854600e-05
            7.34677918993378642475988726801006e-01 2.64565194780213730485485257304390e-01 1.19589629032227804869559228961862e-04
            6.70967910446042203886918287025765e-01 3.28093457323708981299148490506923e-01 1.57837138480472694490250318288815e-04
            6.03930213344110788931118349864846e-01 3.94939915500504390433178514285828e-01 1.97770357903992194826409134478240e-04
            5.34869636659861091843026770220604e-01 4.63803482576349956900685356231406e-01 2.36864816679008667816078159695792e-04
            4.65130363340138908156973229779396e-01 5.33343810164849552180044156557415e-01 2.72379118672929992038544977361880e-04
            3.96069786655889322091184112650808e-01 6.02207377240694952114097304729512e-01 3.01561741052138159585593868428077e-04
            3.29032089553957685090779250458581e-01 6.69053835417490638803883484797552e-01 3.21864214340280110123598822369217e-04
            2.65322081006621690590918660745956e-01 7.32582097960985501039488099195296e-01 3.31144168089031693067575412570136e-04
            2.06179798246544199180618761602091e-01 7.91555665101731253940897659049369e-01 3.27842404107201866765575193340965e-04
            1.52756368406658793457353340272675e-01 8.44826693535522310618546271143714e-01 3.11120068774749800391554721556986e-04
            1.06091597010395888300848810104071e-01 8.91358343999692825931902007141616e-01 2.80944999634538650733206477383419e-04
            6.70937111398499097525416345888516e-02 9.30244980318138070529698779864702e-01 2.38120228230910709575629535805774e-04
            3.65216139064131750657793418213259e-02 9.60729864307880321305788129393477e-01 1.84252393535425692920604046065591e-04
            1.49697510822856538759140221372945e-02 9.82220245976488715022867381776450e-01 1.21665789952908013227955708046579e-04
            2.85270725880037989696802469552495e-03 9.94302723421103595846659572998760e-01 5.33419572873506089220711767762140e-05
            9.97147292741199509080729512788821e-01 2.81000294122561722343078827179852e-03 3.52351496340051358104072422305908e-07
            9.85030248917714068568329821573570e-01 1.47456576348203860504781914642081e-02 4.26916334631605038767718576786514e-06
            9.63478386093586713911918195663020e-01 3.59748944371108755890809050015378e-02 1.61261391484784531117485650852927e-05
            9.32906288860150034736307134153321e-01 6.60893349848996752848862001883390e-02 3.95411562016396216589185497181091e-05
            8.93908402989603945165697496122448e-01 1.04503432211228236559108495384862e-01 7.69872051959229295337722165371019e-05
            8.47243631593341484098402816016460e-01 1.50469643595376872946545177001099e-01 1.29517459832407367354908944712122e-04
            7.93820201753455689797078775882255e-01 2.03093337988597605336948959120491e-01 1.96606725159634971016736626481247e-04
            7.34677918993378642475988726801006e-01 2.61350275497318151884940107265720e-01 2.76123137092146471709835919838838e-04
            6.70967910446042203886918287025765e-01 3.24106561075250709791362169198692e-01 3.64433656827639910948807289159390e-04
            6.03930213344110788931118349864846e-01 3.90140720538436458486586388971773e-01 4.56636349574849221156364054152732e-04
            5.34869636659861091843026770220604e-01 4.58167477580123827518576717920951e-01 5.46902409326301955014293376677870e-04
            4.65130363340138908156973229779396e-01 5.26862771337590296560904334910447e-01 6.28902165973734624156155081919906e-04
            3.96069786655889322091184112650808e-01 5.94889528379277554570592201343970e-01 6.96282567645111253283596308705228e-04
            3.29032089553957685090779250458581e-01 6.60923687842463580821572577406187e-01 7.43159396851935933996091687703256e-04
            2.65322081006621690590918660745956e-01 7.23679973420395583616482326760888e-01 7.64586087125260389918413039822553e-04
            2.06179798246544199180618761602091e-01 7.81936910929116546498107709339820e-01 7.56962571307220519721548068758921e-04
            1.52756368406658793457353340272675e-01 8.34560605322336890310452872654423e-01 7.18352001738021737030359670228563e-04
            1.06091597010395888300848810104071e-01 8.80526816706485915275948173075449e-01 6.48680117809674128617758448456243e-04
            6.70937111398499097525416345888516e-02 9.18940913932814407161231429199688e-01 5.49801056799817389197926331689814e-04
            3.65216139064131750657793418213259e-02 9.49055354480603297062657475180458e-01 4.25424422932426478036194161802541e-04
            1.49697510822856538759140221372945e-02 9.70284591282893904562456555140670e-01 2.80916830919659282501249508712249e-04
            2.85270725880037989696802469552495e-03 9.82220245976488604000564919260796e-01 1.23162423899226901964285008084232e-04
            9.97147292741199509080729512788821e-01 2.74852178570655697117941507201522e-03 5.45542222925190226048201922598446e-07
            9.85030248917714068568329821573570e-01 1.44230316129836250160778376994131e-02 6.60990200459456978644224092733772e-06
            9.63478386093586713911918195663020e-01 3.51877856240841099810801040348451e-02 2.49679365339532700624868705174819e-05
            9.32906288860150034736307134153321e-01 6.46433405260520010937597135125543e-02 6.12211682803705432593527335782824e-05
            8.93908402989603945165697496122448e-01 1.02216960665667586094507157667977e-01 1.19198503470659144826832021024643e-04
            8.47243631593341484098402816016460e-01 1.47177459297964718931694960701861e-01 2.00530560189263011572347039646047e-04
            7.93820201753455689797078775882255e-01 1.98649779259681830145822800659516e-01 3.04404184457091509137527385320254e-04
            7.34677918993378642475988726801006e-01 2.55632090403251410748453054111451e-01 4.27518632885124348370214919512478e-04
            6.70967910446042203886918287025765e-01 3.17015306616447778420564418411232e-01 5.64248908602996276581698076313387e-04
            6.03930213344110788931118349864846e-01 3.81604678827647370020770267728949e-01 7.07005396040916449414137634477129e-04
            5.34869636659861091843026770220604e-01 4.48143051794080649052887110883603e-01 8.46763414391945830675401118980972e-04
            4.65130363340138908156973229779396e-01 5.15335334299506064859031084779417e-01 9.73722800077631424169044560557040e-04
            3.96069786655889322091184112650808e-01 5.81873707265939232868845465418417e-01 1.07804718777348897823520346150872e-03
            3.29032089553957685090779250458581e-01 6.46463079477139102024807471025269e-01 1.15062610364247402267023900890308e-03
            2.65322081006621690590918660745956e-01 7.07846295690334970096557754004607e-01 1.18380082934410103302769190491972e-03
            2.06179798246544199180618761602091e-01 7.64828606833904967032822241890244e-01 1.17199741766832064501779875342891e-03
            1.52756368406658793457353340272675e-01 8.16300926795621717424467078672023e-01 1.11221706716610490607577421684482e-03
            1.06091597010395888300848810104071e-01 8.61261425427919280473076923954068e-01 1.00434480089658852847156467191780e-03
            6.70937111398499097525416345888516e-02 8.98835045567534796084885329037206e-01 8.51251360668898527164771827813183e-04
            3.65216139064131750657793418213259e-02 9.28290600469502624747519803349860e-01 6.58680288813750613834130387402865e-04
            1.49697510822856538759140221372945e-02 9.49055354480603408084959937696112e-01 4.34940660076290968542173098754233e-04
            2.85270725880037989696802469552495e-03 9.60729864307880210283485666877823e-01 1.90691122963173274873494023040621e-04
            9.97147292741199509080729512788821e-01 2.66130854201197765071529310887399e-03 7.28140984419258521665048976279078e-07
            9.85030248917714068568329821573570e-01 1.39653749273356345139696088608616e-02 8.82230623091535732540040964932970e-06
            9.63478386093586713911918195663020e-01 3.40712432926152108958461894872016e-02 3.33249694024935889892155560509934e-05
            9.32906288860150034736307134153321e-01 6.25921450653323940827021942823194e-02 8.17125418816189283324064041025281e-05
            8.93908402989603945165697496122448e-01 9.89735180462151670210957377094019e-02 1.59095505372697893723549222855240e-04
            8.47243631593341484098402816016460e-01 1.42507376750009656651840828089917e-01 2.67650263107825864359462508801357e-04
            7.93820201753455689797078775882255e-01 1.92346430420118119641870180203114e-01 4.06291489856547229000266474585601e-04
            7.34677918993378642475988726801006e-01 2.47520637944539240171692995318153e-01 5.70613648449420766441131380730667e-04
            6.70967910446042203886918287025765e-01 3.06956105581683347249821736113518e-01 7.53108995972283866947305508432464e-04
            6.03930213344110788931118349864846e-01 3.69495994798776983270016671667690e-01 9.43647592119656406645100688024286e-04
            5.34869636659861091843026770220604e-01 4.33923041099822159871735038905172e-01 1.13018409981093751609515507539072e-03
            4.65130363340138908156973229779396e-01 4.98983247760327874864572095248150e-01 1.29963813689491281856691617946353e-03
            3.96069786655889322091184112650808e-01 5.63410294061372995955139231227804e-01 1.43888100236641796984826680017022e-03
            3.29032089553957685090779250458581e-01 6.25950183278466854019939091813285e-01 1.53575285027867801564138705572304e-03
            2.65322081006621690590918660745956e-01 6.85385650915610544764433598174946e-01 1.58003150812609065301028721250987e-03
            2.06179798246544199180618761602091e-01 7.40559858440032026116739416465862e-01 1.56427736951693851716327365863890e-03
            1.52756368406658793457353340272675e-01 7.90398912110140017261983302887529e-01 1.48448790238786413123173169026359e-03
            1.06091597010395888300848810104071e-01 8.33932770813935020370877282402944e-01 1.34050964579782746870229015456744e-03
            6.70937111398499097525416345888516e-02 8.70314143794817751675907402386656e-01 1.13617420925214425257709915939586e-03
            3.65216139064131750657793418213259e-02 8.98835045567534907107187791552860e-01 8.79147559546779072854605363573910e-04
            1.49697510822856538759140221372945e-02 9.18940913932814629205836354230996e-01 5.80519906770517982348700503081318e-04
            2.85270725880037989696802469552495e-03 9.30244980318138181552001242380356e-01 2.54517462003045090088160229413461e-04
            9.97147292741199509080729512788821e-01 2.55005898991119773003277693135260e-03 8.96572665723641283570073332032102e-07
            9.85030248917714068568329821573570e-01 1.33815862831181617287334972843382e-02 1.08630591939426408468844995058866e-05
            9.63478386093586713911918195663020e-01 3.26469775616847618371352268695773e-02 4.10336147692364587038181544809134e-05
            9.32906288860150034736307134153321e-01 5.99756321756691329616373309363553e-02 1.00614074836466636472288860293389e-04
            8.93908402989603945165697496122448e-01 9.48361700541797830732448915114219e-02 1.95897064454372365994647142706242e-04
            8.47243631593341484098402816016460e-01 1.36550201328887704788073165218520e-01 3.29562426797892641175402594910793e-04
            7.93820201753455689797078775882255e-01 1.84305854179287159633204851161281e-01 5.00273782023194437014457225387787e-04
            7.34677918993378642475988726801006e-01 2.37173637710507229092726788621803e-01 7.02606515545368699032779957747152e-04
            6.70967910446042203886918287025765e-01 2.94124549705510829333832134580007e-01 9.27316212859322260009253824364350e-04
            6.03930213344110788931118349864846e-01 3.54050110461999079092265674262308e-01 1.16192970217875667614249035608509e-03
            5.34869636659861091843026770220604e-01 4.15783940275357810634204724919982e-01 1.39161535033512063185001661480555e-03
            4.65130363340138908156973229779396e-01 4.78124462714246134531492771202466e-01 1.60026705515185362245500488853622e-03
            3.96069786655889322091184112650808e-01 5.39858292527604755051129359344486e-01 1.77171921860664813297425723703782e-03
            3.29032089553957685090779250458581e-01 5.99783853284093337876470286573749e-01 1.89099921077127129144523909332065e-03
            2.65322081006621690590918660745956e-01 6.56734765279096466272790166840423e-01 1.94552029274632659085197960990854e-03
            2.06179798246544199180618761602091e-01 7.09602548810316924310370723105734e-01 1.92612194771256536526826685928881e-03
            1.52756368406658793457353340272675e-01 7.57358201660715990577443790243706e-01 1.82787578828557122252729261191462e-03
            1.06091597010395888300848810104071e-01 7.99072232935424286992542874941137e-01 1.65059285533800852510677437123832e-03
            6.70937111398499097525416345888516e-02 8.33932770813934909348574819887290e-01 1.39899107633406649155161094455480e-03
            3.65216139064131750657793418213259e-02 8.61261425427919280473076923954068e-01 1.08250968959802142642379063630642e-03
            1.49697510822856538759140221372945e-02 8.80526816706486026298250635591103e-01 7.14804263811629099489852734450324e-04
            2.85270725880037989696802469552495e-03 8.91358343999692825931902007141616e-01 3.13391780251574933631453223625840e-04
            9.97147292741199509080729512788821e-01 2.41693805781881439209701767367733e-03 1.04755520780146932007068517195281e-06
            9.85030248917714068568329821573570e-01 1.26830262710043335794285468409726e-02 1.26923947899811388718375321227327e-05
            9.63478386093586713911918195663020e-01 3.09427047977194266681344458902458e-02 4.79436620028326771891097335398513e-05
            9.32906288860150034736307134153321e-01 5.68447194832011615250877412108821e-02 1.17557451952874308194127439541887e-04
            8.93908402989603945165697496122448e-01 8.98854299326252437207784851125325e-02 2.28886065689462526659414542962168e-04
            8.47243631593341484098402816016460e-01 1.29421860317867731815510978776729e-01 3.85060631096953182462422349274789e-04
            7.93820201753455689797078775882255e-01 1.74684521027584666041931882318750e-01 5.84519722405268558548807611430220e-04
            7.34677918993378642475988726801006e-01 2.24792443453952606180479278918938e-01 8.20925221717237686876178504036261e-04
            6.70967910446042203886918287025765e-01 2.78770342464440823437854533040081e-01 1.08347595816505400286022808842290e-03
            6.03930213344110788931118349864846e-01 3.35567604410735575992674739609356e-01 1.35759828193529933325345293582131e-03
            5.34869636659861091843026770220604e-01 3.94078738200629719035106290903059e-01 1.62596291771109060717015459829327e-03
            4.65130363340138908156973229779396e-01 4.53164893392711765063296525113401e-01 1.86975150100568377961596233660657e-03
            3.96069786655889322091184112650808e-01 5.11676027182605852594576845149277e-01 2.07007615240573008413837996499751e-03
            3.29032089553957685090779250458581e-01 5.68473289128900827194001976749860e-01 2.20944285602669987511603721941356e-03
            2.65322081006621690590918660745956e-01 6.22451188139388600362167380808387e-01 2.27314527027757664820462046861849e-03
            2.06179798246544199180618761602091e-01 6.72559110565756901323197780584451e-01 2.25048025031910486903896639887535e-03
            1.52756368406658793457353340272675e-01 7.17821771275473419215984449692769e-01 2.13568946995199016036792727391003e-03
            1.06091597010395888300848810104071e-01 7.57358201660716323644351177790668e-01 1.92855214939399071186199030591979e-03
            6.70937111398499097525416345888516e-02 7.90398912110140350328890690434491e-01 1.63458071354281625660320731441288e-03
            3.65216139064131750657793418213259e-02 8.16300926795622050491374466218986e-01 1.26480396535254085120525413543646e-03
            1.49697510822856538759140221372945e-02 8.34560605322337334399662722717039e-01 8.35177066780414196003889948372034e-04
            2.85270725880037989696802469552495e-03 8.44826693535522754707756121206330e-01 3.66166852989810421258043415804195e-04
            9.97147292741199509080729512788821e-01 2.26453665172455338369661426156654e-03 1.17814891063918987023707071054579e-06
            9.85030248917714068568329821573570e-01 1.18832908243392734032317648029675e-02 1.42746950078193613202677536566654e-05
            9.63478386093586713911918195663020e-01 2.89915949195507643976199574353814e-02 5.39205692836340936494737841044866e-05
            9.32906288860150034736307134153321e-01 5.32603433134238232748991492826462e-02 1.32212777831988069444577038602517e-04
            8.93908402989603945165697496122448e-01 8.42176529431389181423739387355454e-02 2.57420197946871471324919999901226e-04
            8.47243631593341484098402816016460e-01 1.21261091187698871740607842184545e-01 4.33064300266350181174007660800385e-04
            7.93820201753455689797078775882255e-01 1.63669689041558596986902784919948e-01 6.57389003529636087135024080652101e-04
            7.34677918993378642475988726801006e-01 2.10618027874322877313062463144888e-01 9.23266046963030963951046814486290e-04
            6.70967910446042203886918287025765e-01 2.61192319713083931365105172517360e-01 1.21854772933154085340357042355208e-03
            6.03930213344110788931118349864846e-01 3.14408197951626111610323732747929e-01 1.52684357352823548738263159663120e-03
            5.34869636659861091843026770220604e-01 3.69229878868327199281651473938837e-01 1.82866394627679228722383353300529e-03
            4.65130363340138908156973229779396e-01 4.24590322885128490515427301943419e-01 2.10284448749866337974312457959059e-03
            3.96069786655889322091184112650808e-01 4.79412003801829467164452580618672e-01 2.32814265609491803654673702794753e-03
            3.29032089553957685090779250458581e-01 5.32627882040371924965427297138376e-01 2.48488354080210211566326350407508e-03
            2.65322081006621690590918660745956e-01 5.83202173879132534928260156448232e-01 2.55652743068573691867517716502789e-03
            2.06179798246544199180618761602091e-01 6.30150512711897148321327222220134e-01 2.53103687097601956837444880932253e-03
            1.52756368406658793457353340272675e-01 6.72559110565756568256290393037489e-01 2.40193567245802243326213698537686e-03
            1.06091597010395888300848810104071e-01 7.09602548810316813288068260590080e-01 2.16897553178887602526314637430005e-03
            6.70937111398499097525416345888516e-02 7.40559858440031915094436953950208e-01 1.83835608154149770382401740675959e-03
            3.65216139064131750657793418213259e-02 7.64828606833904967032822241890244e-01 1.42248103284178394230263808850623e-03
            1.49697510822856538759140221372945e-02 7.81936910929116657520410171855474e-01 9.39294601458997972277809029861828e-04
            2.85270725880037989696802469552495e-03 7.91555665101731253940897659049369e-01 4.11815125111652808719692187011674e-04
            9.97147292741199509080729512788821e-01 2.09582103239285043663753249632009e-03 1.28581157483451368402928814693720e-06
            9.85030248917714068568329821573570e-01 1.09979455729827466092807242148410e-02 1.55791580355733982996176145707778e-05
            9.63478386093586713911918195663020e-01 2.68316233030433115569479696205235e-02 5.88479872794246436731348259829844e-05
            9.32906288860150034736307134153321e-01 4.92922680777678826058441075019800e-02 1.44294764899593639870256644996971e-04
            8.93908402989603945165697496122448e-01 7.79431537142819264030180193003616e-02 2.80944002177706459756623136314602e-04
            8.47243631593341484098402816016460e-01 1.12226730853989764558065189703484e-01 4.72638971951326873795207195172452e-04
            7.93820201753455689797078775882255e-01 1.51475745114245829991972414063639e-01 7.17463117161257473569646592181925e-04
            7.34677918993378642475988726801006e-01 1.94926274336937210840403622569283e-01 1.00763677589168218831638945687246e-03
            6.70967910446042203886918287025765e-01 2.41732610835544758920434560423018e-01 1.32990215473929277063824816451643e-03
            6.03930213344110788931118349864846e-01 2.90983726636500128481799265500740e-01 1.66637096726522399708914079496935e-03
            5.34869636659861091843026770220604e-01 3.41721007399367338042139863318880e-01 1.99577256098261868882959291227053e-03
            4.65130363340138908156973229779396e-01 3.92956911594011304433848863482126e-01 2.29500851521034429647549757191882e-03
            3.96069786655889322091184112650808e-01 4.43694192356878402971886998784612e-01 2.54089508383851039874379296179541e-03
            3.29032089553957685090779250458581e-01 4.92945308157833994577856628893642e-01 2.71195940515334002821057346466205e-03
            2.65322081006621690590918660745956e-01 5.39751644656441209590980179200415e-01 2.79015032146846721911037647601006e-03
            2.06179798246544199180618761602091e-01 5.83202173879132867995167543995194e-01 2.76233036048748829133225868304180e-03
            1.52756368406658793457353340272675e-01 6.22451188139388600362167380808387e-01 2.62143152004346707828474904999894e-03
            1.06091597010395888300848810104071e-01 6.56734765279096799339697554387385e-01 2.36718280611395836404708958866649e-03
            6.70937111398499097525416345888516e-02 6.85385650915610766809038523206254e-01 2.00635039167590444389444037653902e-03
            3.65216139064131750657793418213259e-02 7.07846295690335414185767604067223e-01 1.55247147494979686847826894080526e-03
            1.49697510822856538759140221372945e-02 7.23679973420396138727994639339158e-01 1.02513006618178527731832794245292e-03
            2.85270725880037989696802469552495e-03 7.32582097960985834106395486742258e-01 4.49447985546414397611814095512273e-04
            9.97147292741199509080729512788821e-01 1.91407502855162232890950946284647e-03 1.36844753124817709864812550341062e-06
            9.85030248917714068568329821573570e-01 1.00442226035788069460341631611300e-02 1.65803923140522707917361711071891e-05
            9.63478386093586713911918195663020e-01 2.45048309689032024760724226553066e-02 6.26300030949847330325602756673220e-05
            9.32906288860150034736307134153321e-01 4.50177271675755164004328889859607e-02 1.53568235551385882248220937640326e-04
            8.93908402989603945165697496122448e-01 7.11840571619490186661138864110399e-02 2.98999584172000636931448847022352e-04
            8.47243631593341484098402816016460e-01 1.02494621317141487715218772791559e-01 5.03014319513892752683270614966204e-04
            7.93820201753455689797078775882255e-01 1.38340028405670389943793452403042e-01 7.63572712096097921952164711001387e-04
            7.34677918993378642475988726801006e-01 1.78022602288208264820212889389950e-01 1.07239512021138237371498114214319e-03
            6.70967910446042203886918287025765e-01 2.20769973597714136426262143686472e-01 1.41537170458964109247268492453031e-03
            6.03930213344110788931118349864846e-01 2.65750117143311703760133468676941e-01 1.77346454249427388193127708149177e-03
            5.34869636659861091843026770220604e-01 3.12087547975341383921232818465796e-01 2.12403596877016416152872402278717e-03
            4.65130363340138908156973229779396e-01 3.58880362470700819965685468559968e-01 2.44250308388874225531961492663413e-03
            3.96069786655889322091184112650808e-01 4.05217793302730389104482355833170e-01 2.70419217923663964681102456211192e-03
            3.29032089553957685090779250458581e-01 4.50197936848328206238534221483860e-01 2.88625038494073156078445485661632e-03
            2.65322081006621690590918660745956e-01 4.92945308157833717022100472604507e-01 2.96946643968128663748640150288338e-03
            2.06179798246544199180618761602091e-01 5.32627882040371924965427297138376e-01 2.93985855803755458809378708906479e-03
            1.52756368406658793457353340272675e-01 5.68473289128900494127094589202898e-01 2.78990449467786871470820209140129e-03
            1.06091597010395888300848810104071e-01 5.99783853284093226854167824058095e-01 2.51931583945858581299881429060861e-03
            6.70937111398499097525416345888516e-02 6.25950183278466742997636629297631e-01 2.13529361069958196345242029678957e-03
            3.65216139064131750657793418213259e-02 6.46463079477138991002505008509615e-01 1.65224500915049665218659047383198e-03
            1.49697510822856538759140221372945e-02 6.60923687842463580821572577406187e-01 1.09101266136548205598622018897004e-03
            2.85270725880037989696802469552495e-03 6.69053835417490638803883484797552e-01 4.78332905289497595235914673139632e-04
            9.97147292741199509080729512788821e-01 1.72283610341567396985473781256815e-03 1.42444830324424032089050771304750e-06
            9.85030248917714068568329821573570e-01 9.04068496483321007439748484557640e-03 1.72589092088415157891841816839573e-05
            9.63478386093586713911918195663020e-01 2.20565060781713860593722387193338e-02 6.51930012687151514642197591697936e-05
            9.32906288860150034736307134153321e-01 4.05199192827377652204035030081286e-02 1.59852685300881783899556998207458e-04
            8.93908402989603945165697496122448e-01 6.40719208165059089798987201902492e-02 3.11235499073948926503085932182557e-04
            8.47243631593341484098402816016460e-01 9.22541861615048669476024656432855e-02 5.23599098670283883728837626136965e-04
            7.93820201753455689797078775882255e-01 1.24518209542281224622151114544977e-01 7.94820283066909440486402793624165e-04
            7.34677918993378642475988726801006e-01 1.60236020987232274936928888564580e-01 1.11628058402735456586252205113396e-03
            6.70967910446042203886918287025765e-01 1.98712420041380344315484762773849e-01 1.47329274745644956019774429734071e-03
            6.03930213344110788931118349864846e-01 2.39198510754247611886924573809665e-01 1.84603976457585255629978426838989e-03
            5.34869636659861091843026770220604e-01 2.80906279564833882833596589989611e-01 2.21095757247245696536230852302651e-03
            4.65130363340138908156973229779396e-01 3.23023933779276906097521759875235e-01 2.54245726932672859096906670117733e-03
            3.96069786655889322091184112650808e-01 3.64731702589863093777466929168440e-01 2.81485542806784701588829200602504e-03
            3.29032089553957685090779250458581e-01 4.05217793302730555637936049606651e-01 3.00436397427447004007672326508782e-03
            2.65322081006621690590918660745956e-01 4.43694192356878291949584536268958e-01 3.09098546707642233069557313740461e-03
            2.06179798246544199180618761602091e-01 4.79412003801829633697906274392153e-01 3.06016594655625882859251163381487e-03
            1.52756368406658793457353340272675e-01 5.11676027182605741572274382633623e-01 2.90407533567075800198797885798285e-03
            1.06091597010395888300848810104071e-01 5.39858292527604977095734284375794e-01 2.62241342171145187106562346457395e-03
            6.70937111398499097525416345888516e-02 5.63410294061373106977441693743458e-01 2.22267590918520186252238524104996e-03
            3.65216139064131750657793418213259e-02 5.81873707265939454913450390449725e-01 1.71985958254570418107798346341042e-03
            1.49697510822856538759140221372945e-02 5.94889528379277776615197126375278e-01 1.13566000801107740321438388519937e-03
            2.85270725880037989696802469552495e-03 6.02207377240695174158702229760820e-01 4.97907650652879463400168891240583e-04
            9.97147292741199509080729512788821e-01 1.52582649501156698487736029790085e-03 1.45272387087859875227458059804286e-06
            9.85030248917714068568329821573570e-01 8.00686532227086814050842633605498e-03 1.76015018136544791344921118136924e-05
            9.63478386093586713911918195663020e-01 1.95343023603549749422914061369738e-02 6.64870946468054777218903428348540e-05
            9.32906288860150034736307134153321e-01 3.58863888995332586118180984158244e-02 1.63025791270726097981907942724433e-04
            8.93908402989603945165697496122448e-01 5.67451739456149434337461912036815e-02 3.17413582465417482871006260580771e-04
            8.47243631593341484098402816016460e-01 8.17047432671493267797302451072028e-02 5.33992639589966010781052485612008e-04
            7.93820201753455689797078775882255e-01 1.10279313774732615205387276091642e-01 8.10597615680408271898738092886560e-04
            7.34677918993378642475988726801006e-01 1.41912725065849787409177906738478e-01 1.13843896427934373680179724885875e-03
            6.70967910446042203886918287025765e-01 1.75989274189160299632916917289549e-01 1.50253788652610640111384476824696e-03
            6.03930213344110788931118349864846e-01 2.11845702880584157989929394716455e-01 1.88268400227833499274709971871289e-03
            5.34869636659861091843026770220604e-01 2.48784108439209283858417620649561e-01 2.25484549752720227558389076705225e-03
            4.65130363340138908156973229779396e-01 2.86085528220651807984609149571043e-01 2.59292552592304235969944414819111e-03
            3.96069786655889322091184112650808e-01 3.23023933779276850586370528617408e-01 2.87073083952082887954082579540227e-03
            3.29032089553957685090779250458581e-01 3.58880362470700875476836699817795e-01 3.06400116613278483690407938411226e-03
            2.65322081006621690590918660745956e-01 3.92956911594011137900395169708645e-01 3.15234211191364310095108791642815e-03
            2.06179798246544199180618761602091e-01 4.24590322885128546026578533201246e-01 3.12091081809504414246170256319601e-03
            1.52756368406658793457353340272675e-01 4.53164893392711598529842831339920e-01 2.96172177912680479813789524712320e-03
            1.06091597010395888300848810104071e-01 4.78124462714246190042644002460293e-01 2.67446882302153909893860372903873e-03
            6.70937111398499097525416345888516e-02 4.98983247760327874864572095248150e-01 2.26679644543512255616679595959795e-03
            3.65216139064131750657793418213259e-02 5.15335334299506175881333547295071e-01 1.75399911982277757760417813415188e-03
            1.49697510822856538759140221372945e-02 5.26862771337590407583206797426101e-01 1.15820307348633399872894766957643e-03
            2.85270725880037989696802469552495e-03 5.33343810164849552180044156557415e-01 5.07791211481050937075198969949952e-04
            9.97147292741199509080729512788821e-01 1.32688076378892393439312691327814e-03 1.45272387087860743436228995462134e-06
            9.85030248917714068568329821573570e-01 6.96288576001511880231298334820167e-03 1.76015018136545841665775713469344e-05
            9.63478386093586713911918195663020e-01 1.69873115460582556346391669421791e-02 6.64870946468058707451778688302113e-05
            9.32906288860150034736307134153321e-01 3.12073222403167656324729506422955e-02 1.63025791270727073763863179678424e-04
            8.93908402989603945165697496122448e-01 4.93464230647811114005563126738707e-02 3.17413582465419380224808110213530e-04
            8.47243631593341484098402816016460e-01 7.10516251395091891218669388763374e-02 5.33992639589969154967352693574867e-04
            7.93820201753455689797078775882255e-01 9.59004844718116949975339480261027e-02 8.10597615680413042388297029106070e-04
            7.34677918993378642475988726801006e-01 1.23409355940771556237045558646059e-01 1.13843896427935045885526665898624e-03
            6.70967910446042203886918287025765e-01 1.53042815364797551991316026942513e-01 1.50253788652611529157165914938332e-03
            6.03930213344110788931118349864846e-01 1.84224083775305053078952255418699e-01 1.88268400227834626844969356795900e-03
            5.34869636659861091843026770220604e-01 2.16346254900929624298555609129835e-01 2.25484549752721571969082958730723e-03
            4.65130363340138908156973229779396e-01 2.48784108439209283858417620649561e-01 2.59292552592305797221072793945496e-03
            3.96069786655889322091184112650808e-01 2.80906279564833827322445358731784e-01 2.87073083952084579309471656927144e-03
            3.29032089553957685090779250458581e-01 3.12087547975341439432384049723623e-01 3.06400116613280305150057714058676e-03
            2.65322081006621690590918660745956e-01 3.41721007399367171508686169545399e-01 3.15234211191366218290932366130619e-03
            2.06179798246544199180618761602091e-01 3.69229878868327254792802705196664e-01 3.12091081809506279073906931387228e-03
            1.52756368406658793457353340272675e-01 3.94078738200629552501652597129578e-01 2.96172177912682257905352400939591e-03
            1.06091597010395888300848810104071e-01 4.15783940275357866145355956177809e-01 2.67446882302155514513075651450436e-03
            6.70937111398499097525416345888516e-02 4.33923041099822159871735038905172e-01 2.26679644543513600027373477985293e-03
            3.65216139064131750657793418213259e-02 4.48143051794080704564038342141430e-01 1.75399911982278798594503399499445e-03
            1.49697510822856538759140221372945e-02 4.58167477580123938540879180436605e-01 1.15820307348634093762285157680481e-03
            2.85270725880037989696802469552495e-03 4.63803482576350012411836587489233e-01 5.07791211481053972841281929362367e-04
            9.97147292741199509080729512788821e-01 1.12987115538481738309661839281262e-03 1.42444830324423714451695550942123e-06
            9.85030248917714068568329821573570e-01 5.92906611745277947050913880389089e-03 1.72589092088414785197345024947424e-05
            9.63478386093586713911918195663020e-01 1.44651078282418479870052863134333e-02 6.51930012687150023864210424129340e-05
            9.32906288860150034736307134153321e-01 2.65737918571122624933344980036054e-02 1.59852685300881404428796628280907e-04
            8.93908402989603945165697496122448e-01 4.20196761938901527932976875945315e-02 3.11235499073948221771673816604675e-04
            8.47243631593341484098402816016460e-01 6.05021822451536697706764300619398e-02 5.23599098670282691106447892082087e-04
            7.93820201753455689797078775882255e-01 8.16615887042631133363457252016815e-02 7.94820283066907705762926816817071e-04
            7.34677918993378642475988726801006e-01 1.05086060019389110342658000263327e-01 1.11628058402735196377730808592332e-03
            6.70967910446042203886918287025765e-01 1.30319669512577562819899412716040e-01 1.47329274745644630759122684082740e-03
            6.03930213344110788931118349864846e-01 1.56871275901641626937532691954402e-01 1.84603976457584843633152882347304e-03
            5.34869636659861091843026770220604e-01 1.84224083775305108590103486676526e-01 2.21095757247245219487274958680700e-03
            4.65130363340138908156973229779396e-01 2.11845702880584213501080625974282e-01 2.54245726932672295311776977655427e-03
            3.96069786655889322091184112650808e-01 2.39198510754247639642500189438579e-01 2.81485542806784094435612608720021e-03
            3.29032089553957685090779250458581e-01 2.65750117143311814782435931192595e-01 3.00436397427446353486368835206122e-03
            2.65322081006621690590918660745956e-01 2.90983726636500072970648034242913e-01 3.09098546707641539180166923017623e-03
            2.06179798246544199180618761602091e-01 3.14408197951626278143777426521410e-01 3.06016594655625232337947672078826e-03
            1.52756368406658793457353340272675e-01 3.35567604410735520481523508351529e-01 2.90407533567075149677494394495625e-03
            1.06091597010395888300848810104071e-01 3.54050110461999190114568136777962e-01 2.62241342171144623321432653995089e-03
            6.70937111398499097525416345888516e-02 3.69495994798777094292319134183344e-01 2.22267590918519709203282630483045e-03
            3.65216139064131750657793418213259e-02 3.81604678827647481043072730244603e-01 1.71985958254570049479059701269534e-03
            1.49697510822856538759140221372945e-02 3.90140720538436680531191314003081e-01 1.13566000801107480112916991998873e-03
            2.85270725880037989696802469552495e-03 3.94939915500504556966632208059309e-01 4.97907650652878379197996405736149e-04
            9.97147292741199509080729512788821e-01 9.38632230248868265100326002681186e-04 1.36844753124817646337341506268537e-06
            9.85030248917714068568329821573570e-01 4.92552847870717826206377054631957e-03 1.65803923140522640154725930727864e-05
            9.63478386093586713911918195663020e-01 1.20167829375100246314111984702322e-02 6.26300030949847059275059635297112e-05
            9.32906288860150034736307134153321e-01 2.20759839722745009049642561649307e-02 1.53568235551385800933058001227494e-04
            8.93908402989603945165697496122448e-01 3.49075398484470222904008096520556e-02 2.98999584172000528511231598471909e-04
            8.47243631593341484098402816016460e-01 5.02617470895170143085906033775245e-02 5.03014319513892535842836117865318e-04
            7.93820201753455689797078775882255e-01 6.78397698408738925035521560857887e-02 7.63572712096097596691512965350057e-04
            7.34677918993378642475988726801006e-01 8.72994787184130510704349603656738e-02 1.07239512021138194003411214794141e-03
            6.70967910446042203886918287025765e-01 1.08262115956243687442395184916677e-01 1.41537170458964044195138143322765e-03
            6.03930213344110788931118349864846e-01 1.30319669512577451797596950200386e-01 1.77346454249427323140997359018911e-03
            5.34869636659861091843026770220604e-01 1.53042815364797468724589180055773e-01 2.12403596877016329416698603438363e-03
            4.65130363340138908156973229779396e-01 1.75989274189160216366190070402808e-01 2.44250308388874138795787693823058e-03
            3.96069786655889322091184112650808e-01 1.98712420041380205537606684629282e-01 2.70419217923663877944928657370838e-03
            3.29032089553957685090779250458581e-01 2.20769973597714053159535296799731e-01 2.88625038494073025974184787401100e-03
            2.65322081006621690590918660745956e-01 2.41732610835544509120254019762797e-01 2.96946643968128533644379452027806e-03
            2.06179798246544199180618761602091e-01 2.61192319713083820342802710001706e-01 2.93985855803755328705118010645947e-03
            1.52756368406658793457353340272675e-01 2.78770342464440545882098376750946e-01 2.78990449467786784734646410299774e-03
            1.06091597010395888300848810104071e-01 2.94124549705510718311529672064353e-01 2.51931583945858451195620730800329e-03
            6.70937111398499097525416345888516e-02 3.06956105581683180716368042340036e-01 2.13529361069958109609068230838602e-03
            3.65216139064131750657793418213259e-02 3.17015306616447667398261955895578e-01 1.65224500915049600166528698252932e-03
            1.49697510822856538759140221372945e-02 3.24106561075250598769059706683038e-01 1.09101266136548162230535119476826e-03
            2.85270725880037989696802469552495e-03 3.28093457323708870276846027991269e-01 4.78332905289497378395480176038745e-04
            9.97147292741199509080729512788821e-01 7.56886226407641566835127200363331e-04 1.28581157483452956589704916506856e-06
            9.85030248917714068568329821573570e-01 3.97180550930324553771111339983690e-03 1.55791580355735914231295885512552e-05
            9.63478386093586713911918195663020e-01 9.68999060336993289777041127308621e-03 5.88479872794253755096012536984773e-05
            9.32906288860150034736307134153321e-01 1.78014430620821624551286532778249e-02 1.44294764899595401698786933941676e-04
            8.93908402989603945165697496122448e-01 2.81484432961141665952009560669467e-02 2.80944002177709929203575089928790e-04
            8.47243631593341484098402816016460e-01 4.05296375526687999157893216306547e-02 4.72638971951332728486938616896396e-04
            7.93820201753455689797078775882255e-01 5.47040531322985495998878491263895e-02 7.17463117161266364027460973318284e-04
            7.34677918993378642475988726801006e-01 7.03958066696842299503344975164509e-02 1.00763677589169454822115579162300e-03
            6.70967910446042203886918287025765e-01 8.72994787184132037261008463246981e-02 1.32990215473930925051126994418382e-03
            6.03930213344110788931118349864846e-01 1.05086060019389207487172654964525e-01 1.66637096726524459693041801955360e-03
            5.34869636659861091843026770220604e-01 1.23409355940771708892711444605084e-01 1.99577256098264340863912558177162e-03
            4.65130363340138908156973229779396e-01 1.41912725065849981698207216140872e-01 2.29500851521037291941285118923588e-03
            3.96069786655889322091184112650808e-01 1.60236020987232441470382582338061e-01 2.54089508383854162376636054432311e-03
            3.29032089553957685090779250458581e-01 1.78022602288208514620393430050171e-01 2.71195940515337385531835501240039e-03
            2.65322081006621690590918660745956e-01 1.94926274336937377373857316342765e-01 2.79015032146850147989902701795017e-03
            2.06179798246544199180618761602091e-01 2.10618027874323182624394235062937e-01 2.76233036048752255212090922498192e-03
            1.52756368406658793457353340272675e-01 2.24792443453952800469508588321332e-01 2.62143152004349960434992361513196e-03
            1.06091597010395888300848810104071e-01 2.37173637710507562159634176168765e-01 2.36718280611398785434618119438710e-03
            6.70937111398499097525416345888516e-02 2.47520637944539545483024767236202e-01 2.00635039167592916370397304604012e-03
            3.65216139064131750657793418213259e-02 2.55632090403251799326511672916240e-01 1.55247147494981595043650468568330e-03
            1.49697510822856538759140221372945e-02 2.61350275497318540462998726070509e-01 1.02513006618179785406352877430436e-03
            2.85270725880037989696802469552495e-03 2.64565194780214063552392644851352e-01 4.49447985546419981253002395860108e-04
            9.97147292741199509080729512788821e-01 5.88170607075937318733438452511564e-04 1.17814891063919347012709654132223e-06
            9.85030248917714068568329821573570e-01 3.08646025794671137118529991028026e-03 1.42746950078194036719151163716823e-05
            9.63478386093586713911918195663020e-01 7.53001898686246270986366369015741e-03 5.39205692836342495035360788957490e-05
            9.32906288860150034736307134153321e-01 1.38333678264261922957745198914381e-02 1.32212777831988476020391720666680e-04
            8.93908402989603945165697496122448e-01 2.18739440672571297530346612347785e-02 2.57420197946872230266440739754330e-04
            8.47243631593341484098402816016460e-01 3.14952772189596302832015339845384e-02 4.33064300266351482216614643405705e-04
            7.93820201753455689797078775882255e-01 4.25101092049856923993367274761113e-02 6.57389003529638038698934554560083e-04
            7.34677918993378642475988726801006e-01 5.47040531322984455164792905179638e-02 9.23266046963033674456478028247375e-04
            6.70967910446042203886918287025765e-01 6.78397698408738925035521560857887e-02 1.21854772933154453969095687426716e-03
            6.03930213344110788931118349864846e-01 8.16615887042630439474066861293977e-02 1.52684357352823982419132153864894e-03
            5.34869636659861091843026770220604e-01 9.59004844718116394863827167682757e-02 1.82866394627679770823469596052746e-03
            4.65130363340138908156973229779396e-01 1.10279313774732559694236044833815e-01 2.10284448749866945127529049841542e-03
            3.96069786655889322091184112650808e-01 1.24518209542281141355424267658236e-01 2.32814265609492497544064093517591e-03
            3.29032089553957685090779250458581e-01 1.38340028405670334432642221145215e-01 2.48488354080210948823803640550523e-03
            2.65322081006621690590918660745956e-01 1.51475745114245691214094335919071e-01 2.55652743068574429124995006645804e-03
            2.06179798246544199180618761602091e-01 1.63669689041558541475751553662121e-01 2.53103687097602694094922171075268e-03
            1.52756368406658793457353340272675e-01 1.74684521027584499508478188545268e-01 2.40193567245802980583690988680701e-03
            1.06091597010395888300848810104071e-01 1.84305854179287104122053619903454e-01 2.16897553178888209679531229312488e-03
            6.70937111398499097525416345888516e-02 1.92346430420118008619567717687460e-01 1.83835608154150312483487983428176e-03
            3.65216139064131750657793418213259e-02 1.98649779259681746879095953772776e-01 1.42248103284178806227089353342308e-03
            1.49697510822856538759140221372945e-02 2.03093337988597549825797727862664e-01 9.39294601459000682783240243622913e-04
            2.85270725880037989696802469552495e-03 2.05591627639468282895407469368365e-01 4.11815125111654001342081921066551e-04
            9.97147292741199509080729512788821e-01 4.35769200981677502955424774455651e-04 1.04755520780147249644423737557908e-06
            9.85030248917714068568329821573570e-01 2.28672481128165726652068379110005e-03 1.26923947899811761412872113119477e-05
            9.63478386093586713911918195663020e-01 5.57890910869381258241350707294259e-03 4.79436620028328127143812942279055e-05
            9.32906288860150034736307134153321e-01 1.02489916566488765969911156616945e-02 1.17557451952874647007306341262023e-04
            8.93908402989603945165697496122448e-01 1.62061670777708388690996343939332e-02 2.28886065689463204285772346402439e-04
            8.47243631593341484098402816016460e-01 2.33345080887908222500026766965675e-02 3.85060631096954320874703459054444e-04
            7.93820201753455689797078775882255e-01 3.14952772189597066110344769640506e-02 5.84519722405270184852066339686871e-04
            7.34677918993378642475988726801006e-01 4.05296375526688207324710333523399e-02 8.20925221717240072120957972146016e-04
            6.70967910446042203886918287025765e-01 5.02617470895171322697869698004070e-02 1.08347595816505725546674554493620e-03
            6.03930213344110788931118349864846e-01 6.05021822451537599762971808559087e-02 1.35759828193530323638127388363728e-03
            5.34869636659861091843026770220604e-01 7.10516251395093140219572092064482e-02 1.62596291771109516081927903741189e-03
            4.65130363340138908156973229779396e-01 8.17047432671494794353961310662271e-02 1.86975150100568920062682476412874e-03
            3.96069786655889322091184112650808e-01 9.22541861615050196032683516023098e-02 2.07007615240573615567054588382234e-03
            3.29032089553957685090779250458581e-01 1.02494621317141682004248082193953e-01 2.20944285602670638032907213244016e-03
            2.65322081006621690590918660745956e-01 1.12226730853989917213731075662508e-01 2.27314527027758315341765538164509e-03
            2.06179798246544199180618761602091e-01 1.21261091187699107663000575030310e-01 2.25048025031911137425200131190195e-03
            1.52756368406658793457353340272675e-01 1.29421860317867926104540288179123e-01 2.13568946995199623190009319273486e-03
            1.06091597010395888300848810104071e-01 1.36550201328887982343829321507656e-01 1.92855214939399634971328723054285e-03
            6.70937111398499097525416345888516e-02 1.42507376750009934207596984379052e-01 1.63458071354282102709276625063239e-03
            3.65216139064131750657793418213259e-02 1.47177459297964996487451116990997e-01 1.26480396535254453749264058615154e-03
            1.49697510822856538759140221372945e-02 1.50469643595377178257876948919147e-01 8.35177066780416581248669416481789e-04
            2.85270725880037989696802469552495e-03 1.52320599205677142951032010387280e-01 3.66166852989811451250107277033408e-04
            9.97147292741199509080729512788821e-01 3.02648268889292809766949909899836e-04 8.96572665723638424833876348768458e-07
            9.85030248917714068568329821573570e-01 1.58816479916782261200269843470778e-03 1.08630591939426052715007148252724e-05
            9.63478386093586713911918195663020e-01 3.87463634472846266826318029075082e-03 4.10336147692363231785465937928592e-05
            9.32906288860150034736307134153321e-01 7.11807896418087740486591030730779e-03 1.00614074836466311211637114642059e-04
            8.93908402989603945165697496122448e-01 1.12554269562162596179932805284807e-02 1.95897064454371742578397963541192e-04
            8.47243631593341484098402816016460e-01 1.62061670777707868273953550897204e-02 3.29562426797891556973230109406359e-04
            7.93820201753455689797078775882255e-01 2.18739440672571089363529495130933e-02 5.00273782023192810711198497131136e-04
            7.34677918993378642475988726801006e-01 2.81484432961140867979210611338203e-02 7.02606515545366422208217738187841e-04
            6.70967910446042203886918287025765e-01 3.49075398484469875959312901159137e-02 9.27316212859319332663388113502378e-04
            6.03930213344110788931118349864846e-01 4.20196761938900764654647446150193e-02 1.16192970217875277301466940826913e-03
            5.34869636659861091843026770220604e-01 4.93464230647810350727233696943586e-02 1.39161535033511629504132667278782e-03
            4.65130363340138908156973229779396e-01 5.67451739456148532281254404097126e-02 1.60026705515184841828457695811494e-03
            3.96069786655889322091184112650808e-01 6.40719208165057979575962576745951e-02 1.77171921860664249512296031241476e-03
            3.29032089553957685090779250458581e-01 7.11840571619489215215992317098426e-02 1.89099921077126521991307317449582e-03
            2.65322081006621690590918660745956e-01 7.79431537142817737473521333413373e-02 1.94552029274632030247937919398282e-03
            2.06179798246544199180618761602091e-01 8.42176529431387932422836684054346e-02 1.92612194771255929373610094046398e-03
            1.52756368406658793457353340272675e-01 8.98854299326250633095369835245947e-02 1.82787578828556515099512669308979e-03
            1.06091597010395888300848810104071e-01 9.48361700541796442953668133668543e-02 1.65059285533800332093634644081703e-03
            6.70937111398499097525416345888516e-02 9.89735180462150143654298517503776e-02 1.39899107633406193790248650543617e-03
            3.65216139064131750657793418213259e-02 1.02216960665667433438841271708952e-01 1.08250968959801795697683868269223e-03
            1.49697510822856538759140221372945e-02 1.04503432211228097781230417240295e-01 7.14804263811626822665290514891012e-04
            2.85270725880037989696802469552495e-03 1.05788948741506613759888466574921e-01 3.13391780251573903639389362396628e-04
            9.97147292741199509080729512788821e-01 1.91398716788513187240031165892162e-04 7.28140984419260427489180298454841e-07
            9.85030248917714068568329821573570e-01 1.00437615495035156149006283499148e-03 8.82230623091537934825703826113852e-06
            9.63478386093586713911918195663020e-01 2.45037061379801491059482465573183e-03 3.33249694024936770806420704982287e-05
            9.32906288860150034736307134153321e-01 4.50156607451762755950364081058979e-03 8.17125418816191316203137451346095e-05
            8.93908402989603945165697496122448e-01 7.11807896418087740486591030730779e-03 1.59095505372698300299363904919403e-04
            8.47243631593341484098402816016460e-01 1.02489916566488419025215961255526e-02 2.67650263107826514880766000104018e-04
            7.93820201753455689797078775882255e-01 1.38333678264261766832632361001743e-02 4.06291489856548313202438960090035e-04
            7.34677918993378642475988726801006e-01 1.78014430620821104134243739736121e-02 5.70613648449422175903955611886431e-04
            6.70967910446042203886918287025765e-01 2.20759839722744766188355924896314e-02 7.53108995972285818511215982340445e-04
            6.03930213344110788931118349864846e-01 2.65737918571122104516302186993926e-02 9.43647592119658900310097404684484e-04
            5.34869636659861091843026770220604e-01 3.12073222403167135907686713380826e-02 1.13018409981094055186123803480314e-03
            4.65130363340138908156973229779396e-01 3.58863888995331961617729632507690e-02 1.29963813689491607117343363597683e-03
            3.96069786655889322091184112650808e-01 4.05199192827376888925705600286165e-02 1.43888100236642165613565325088530e-03
            3.29032089553957685090779250458581e-01 4.50177271675754470114938499136770e-02 1.53575285027868191876920800353901e-03
            2.65322081006621690590918660745956e-01 4.92922680777677785224355488935544e-02 1.58003150812609477297854265742671e-03
            2.06179798246544199180618761602091e-01 5.32603433134237400081723023959057e-02 1.56427736951694263713152910355575e-03
            1.52756368406658793457353340272675e-01 5.68447194832010435638913747879997e-02 1.48448790238786803435955263807955e-03
            1.06091597010395888300848810104071e-01 5.99756321756690427560165801423864e-02 1.34050964579783093814924210818162e-03
            6.70937111398499097525416345888516e-02 6.25921450653322830603997317666654e-02 1.13617420925214728834318211880827e-03
            3.65216139064131750657793418213259e-02 6.46433405260519039492450588113570e-02 8.79147559546781349679167583133221e-04
            1.49697510822856538759140221372945e-02 6.60893349848995781403715454871417e-02 5.80519906770519500231741982787526e-04
            2.85270725880037989696802469552495e-03 6.69023124230614107954551172952051e-02 2.54517462003045740609463720716121e-04
            9.97147292741199509080729512788821e-01 1.04185473093933812565800578475717e-04 5.45542222925193825938227753374887e-07
            9.85030248917714068568329821573570e-01 5.46719469302359541498792516733829e-04 6.60990200459461298512255089665501e-06
            9.63478386093586713911918195663020e-01 1.33382828232911430747786862838211e-03 2.49679365339534326928127433431470e-05
            9.32906288860150034736307134153321e-01 2.45037061379801491059482465573183e-03 6.12211682803709498351674156424451e-05
            8.93908402989603945165697496122448e-01 3.87463634472846310194404928495260e-03 1.19198503470659917320879916946552e-04
            8.47243631593341484098402816016460e-01 5.57890910869379350045527132806455e-03 2.00530560189264339720008334388979e-04
            7.93820201753455689797078775882255e-01 7.53001898686245577096975978292903e-03 3.04404184457093514911546483503457e-04
            7.34677918993378642475988726801006e-01 9.68999060336990514219479564417270e-03 4.27518632885127113085754757548784e-04
            6.70967910446042203886918287025765e-01 1.20167829375100124883468666325825e-02 5.64248908602999962869084527028463e-04
            6.03930213344110788931118349864846e-01 1.44651078282418219661531466613269e-02 7.07005396040921003063262073595752e-04
            5.34869636659861091843026770220604e-01 1.69873115460582313485105032668798e-02 8.46763414391951360106480795053585e-04
            4.65130363340138908156973229779396e-01 1.95343023603549437172688385544461e-02 9.73722800077637820961862225033201e-04
            3.96069786655889322091184112650808e-01 2.20565060781713478954557672295778e-02 1.07804718777349613396954186583798e-03
            3.29032089553957685090779250458581e-01 2.45048309689031677816029031191647e-02 1.15062610364248161208544640743412e-03
            2.65322081006621690590918660745956e-01 2.68316233030432560457967383626965e-02 1.18380082934410862244289930345076e-03
            2.06179798246544199180618761602091e-01 2.89915949195507227642565339920111e-02 1.17199741766832823443300615195994e-03
            1.52756368406658793457353340272675e-01 3.09427047977193676875362626788046e-02 1.11221706716611227865054711827497e-03
            1.06091597010395888300848810104071e-01 3.26469775616847132648778995189787e-02 1.00434480089659503368459958494441e-03
            6.70937111398499097525416345888516e-02 3.40712432926151623235888621366030e-02 8.51251360668904056595851503885797e-04
            3.65216139064131750657793418213259e-02 3.51877856240840614088227766842465e-02 6.58680288813754950642820329420601e-04
            1.49697510822856538759140221372945e-02 3.59748944371108270168235776509391e-02 4.34940660076293787467821561065762e-04
            2.85270725880037989696802469552495e-03 3.64174284333192432860926146531710e-02 1.90691122963174494600938069233109e-04
            9.97147292741199509080729512788821e-01 4.27043175748727878203794827705053e-05 3.52351496340037805576916353500483e-07
            9.85030248917714068568329821573570e-01 2.24093447465597504211429202847228e-04 4.26916334631588606328541843359936e-06
            9.63478386093586713911918195663020e-01 5.46719469302348265796198667487715e-04 1.61261391484778330836311749374445e-05
            9.32906288860150034736307134153321e-01 1.00437615495033096164878561040723e-03 3.95411562016381037758770700119015e-05
            8.93908402989603945165697496122448e-01 1.58816479916778986909708937247387e-03 7.69872051959199750828521935375193e-05
            8.47243631593341484098402816016460e-01 2.28672481128160218905032152747481e-03 1.29517459832402407129969823529336e-04
            7.93820201753455689797078775882255e-01 3.08646025794664458433147480320713e-03 1.96606725159627435811637852225431e-04
            7.34677918993378642475988726801006e-01 3.97180550930315186264341065225381e-03 2.76123137092135900738654186170606e-04
            6.70967910446042203886918287025765e-01 4.92552847870707070920825998427972e-03 3.64433656827625870530673601876970e-04
            6.03930213344110788931118349864846e-01 5.92906611745264589680148858974462e-03 4.56636349574831711291278413256123e-04
            5.34869636659861091843026770220604e-01 6.96288576001496354456188342396672e-03 5.46902409326280921492147157891850e-04
            4.65130363340138908156973229779396e-01 8.00686532227068946399040072492426e-03 6.28902165973710446447708655171027e-04
            3.96069786655889322091184112650808e-01 9.04068496483300711175079555914635e-03 6.96282567645084581910153165296151e-04
            3.29032089553957685090779250458581e-01 1.00442226035785849014292381298219e-02 7.43159396851907311058738070386198e-04
            2.65322081006621690590918660745956e-01 1.09979455729824968091001835546194e-02 7.64586087125231008039538682652392e-04
            2.06179798246544199180618761602091e-01 1.18832908243390114599868923050963e-02 7.56962571307191463103325457240089e-04
            1.52756368406658793457353340272675e-01 1.26830262710040473500550106678020e-02 7.18352001737994089874961289865496e-04
            1.06091597010395888300848810104071e-01 1.33815862831178650910191052503251e-02 6.48680117809649191967791281854261e-04
            6.70937111398499097525416345888516e-02 1.39653749273353239984674090123917e-02 5.49801056799796247255562864353351e-04
            3.65216139064131750657793418213259e-02 1.44230316129833058269582579669077e-02 4.25424422932410160793498254960809e-04
            1.49697510822856538759140221372945e-02 1.47456576348200599224647078244743e-02 2.80916830919648494689633277943130e-04
            2.85270725880037989696802469552495e-03 1.49270467647107819147977991747211e-02 1.23162423899222185684834696139944e-04
            9.97147292741199509080729512788821e-01 8.13793870441269504088354497239877e-06 1.52604324215665168245308199942989e-07
            9.85030248917714068568329821573570e-01 4.27043175748720763127037891582205e-05 1.84898544265609851065357109589637e-06
            9.63478386093586713911918195663020e-01 1.04185473093929922990506786728560e-04 6.98427165067670272945079407156399e-06
            9.32906288860150034736307134153321e-01 1.91398716788506058610747073700509e-04 1.71253747565577693786047913659942e-05
            8.93908402989603945165697496122448e-01 3.02648268889281534064356060653722e-04 3.33433532827631670474553826633013e-05
            8.47243631593341484098402816016460e-01 4.35769200981659776249904636458155e-04 5.60943394228674728691809436842419e-05
            7.93820201753455689797078775882255e-01 5.88170607075914767328250754019336e-04 8.51508699150959274355374151177500e-05
            7.34677918993378642475988726801006e-01 7.56886226407611100754080357688736e-04 1.19589629032226015935974627879546e-04
            6.70967910446042203886918287025765e-01 9.38632230248832269588199483933977e-04 1.57837138480470336350525162316671e-04
            6.03930213344110788931118349864846e-01 1.12987115538477293080754648713082e-03 1.97770357903989240375489111478657e-04
            5.34869636659861091843026770220604e-01 1.32688076378887232636971660326708e-03 2.36864816679005144159017581806381e-04
            4.65130363340138908156973229779396e-01 1.52582649501150757059830809225787e-03 2.72379118672925926280398156720253e-04
            3.96069786655889322091184112650808e-01 1.72283610341560653247960921419235e-03 3.01561741052133660146578053584676e-04
            3.29032089553957685090779250458581e-01 1.91407502855154816948091145434319e-03 3.21864214340275285423931261874486e-04
            2.65322081006621690590918660745956e-01 2.09582103239276760359155460378133e-03 3.31144168089026759947690603524961e-04
            2.06179798246544199180618761602091e-01 2.26453665172446534648020843860650e-03 3.27842404107196933645690384295790e-04
            1.52756368406658793457353340272675e-01 2.41693805781871898230583894928714e-03 3.11120068774745138322213033887920e-04
            1.06091597010395888300848810104071e-01 2.55005898991109885079464625334822e-03 2.80944999634534476554842408191348e-04
            6.70937111398499097525416345888516e-02 2.66130854201187400098760349465010e-03 2.38120228230907131708460333641142e-04
            3.65216139064131750657793418213259e-02 2.74852178570645028568564249837891e-03 1.84252393535422928205064208029285e-04
            1.49697510822856538759140221372945e-02 2.81000294122550880321353972135512e-03 1.21665789952906197189316794826652e-04
            2.85270725880037989696802469552495e-03 2.84456932009596744417856406528244e-03 5.33419572873498093229689687166939e-05
            ];
        
    case  43
        
        % ALG. DEG.:   43
        % PTS CARD.:  529
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.2e-15
        
        xyw=[
            9.97384667498776078531363964430057e-01 2.60849253713201856622050378575750e-03 1.17610178504428544303244004963493e-07
            9.86271235609057494997387038893066e-01 1.36928591072291716557174368062988e-02 1.42644189164332600739412779816684e-06
            9.66485543413008074686842974188039e-01 3.34268051394191051395310410043749e-02 5.39812746813001293117763312867652e-06
            9.38376179135220755966884098597802e-01 6.14626540832220388810291922254692e-02 1.32718639180100048547816110611208e-05
            9.02444200809420005171546108613256e-01 9.73006583382739903553826366078283e-02 2.59330423710481819876347286957241e-05
            8.59330681565975140756563632749021e-01 1.40301421393599329290324817520741e-01 4.38245836222476357587132800297525e-05
            8.09804937881823172496353890892351e-01 1.89697638790646855300181528036774e-01 6.68915078706977203094885542355996e-05
            7.54750738923003927638433196989354e-01 2.44607852713600304461749601614429e-01 9.45631352946293366476884778570877e-05
            6.95150519015145573753500229940983e-01 3.04052198229253500816327004940831e-01 1.25774572418214464425770260369575e-04
            6.32067840485172394160429121257039e-01 3.66969894579802946577018474272336e-01 1.59026298075460247819640224875570e-04
            5.66628412149233096073430715478025e-01 4.32238177051953775809067792579299e-01 1.92477901869551079432063556140520e-04
            5.00000000000000000000000000000000e-01 4.98692333749387983754530750957201e-01 2.24069542746758342465981939106712e-04
            4.33371587850766792904266822006321e-01 5.65146490446822413744598634366412e-01 2.51662663099445622731653582349054e-04
            3.67932159514827383794965953711653e-01 6.30414772918973298487799183931202e-01 2.73190060193302143576321583395838e-04
            3.04849480984854481757651001316844e-01 6.93332469269522522203885728231398e-01 2.86804684767606699827974248151463e-04
            2.45249261076996127872718034268473e-01 7.52776814785175774069614362815628e-01 2.91016559744455382933991938188001e-04
            1.90195062118176771992494877849822e-01 8.07687028708129362009060514537850e-01 2.84807990138008461020180384082323e-04
            1.40669318434024914754587598508806e-01 8.57083246105176721485463531280402e-01 2.67718716012077875832098072805820e-04
            9.75557991905800503396051226445707e-02 9.00084009160502129809344751265598e-01 2.39894746301841876000293662940521e-04
            6.16238208647792440331159014021978e-02 9.35922013415554143733743330813013e-01 2.02097188694489869782711100043571e-04
            3.35144565869919253131570258119609e-02 9.63957862359356987269620731240138e-01 1.55670498368553516296136640661985e-04
            1.37287643909422829580080360756256e-02 9.83691808391547017897948990139412e-01 1.02475253193489365135214474999970e-04
            2.61533250122397697978726682777051e-03 9.94776174961644166216956364223734e-01 4.48518835472044564939217425880713e-05
            9.97384667498776078531363964430057e-01 2.57942721751069871036210479076090e-03 2.71737480088410374893750530087000e-07
            9.86271235609057494997387038893066e-01 1.35402854192404401012517212166131e-02 3.29578383484136319995453506814798e-06
            9.66485543413008074686842974188039e-01 3.30543445088186393920892669484601e-02 1.24723350822092857409333077622904e-05
            9.38376179135220755966884098597802e-01 6.07778019472570985026749212920549e-02 3.06645470912242213801061907485490e-05
            9.02444200809420005171546108613256e-01 9.62164786085224210676614120529848e-02 5.99181097635121539960248449041558e-05
            8.59330681565975140756563632749021e-01 1.38738102504209676046897925516532e-01 1.01256388442476298779315602160978e-04
            8.09804937881823172496353890892351e-01 1.87583918922035713938711865012010e-01 1.54552352689550789128711261177784e-04
            7.54750738923003927638433196989354e-01 2.41882291754617312351527402825013e-01 2.18487450839592204721376078602191e-04
            6.95150519015145573753500229940983e-01 3.00664274285712240786239135559299e-01 2.90601254098393725275317756029381e-04
            6.32067840485172394160429121257039e-01 3.62880905584997859314455581625225e-01 3.67429129488029953404248928450215e-04
            5.66628412149233096073430715478025e-01 4.27421931427435064509978701607906e-01 4.44718821889779575371598463462419e-04
            5.00000000000000000000000000000000e-01 4.93135617804528691987542288188706e-01 5.17711083214398497080732663988556e-04
            4.33371587850766792904266822006321e-01 5.58849304181622485998559568542987e-01 5.81464791335274504280317131588163e-04
            3.67932159514827383794965953711653e-01 6.23390330024059857727536382299149e-01 6.31203689052591237866607354334292e-04
            3.04849480984854481757651001316844e-01 6.85606961323345087677694209560286e-01 6.62660182199840096239906728214919e-04
            2.45249261076996127872718034268473e-01 7.44388943854440210401435251696967e-01 6.72391689346674300241379906850625e-04
            1.90195062118176771992494877849822e-01 7.98687316687021864325402020767797e-01 6.58046833473967273699101188100258e-04
            1.40669318434024914754587598508806e-01 8.47533133104847791194913497747621e-01 6.18562187276056768027443499846640e-04
            9.75557991905800503396051226445707e-02 8.90054757000535046174150011211168e-01 5.54275103358136409306688996423418e-04
            6.16238208647792440331159014021978e-02 9.25493433661800479761438964487752e-01 4.66944115612619094646956741101462e-04
            3.35144565869919253131570258119609e-02 9.53216891100238883360873387573520e-01 3.59675578157420215014150599586173e-04
            1.37287643909422829580080360756256e-02 9.72730950189817256124058530986076e-01 2.36768342913212142316486952253740e-04
            2.61533250122397697978726682777051e-03 9.83691808391546795853344065108104e-01 1.03629957605048960995325346701179e-04
            9.97384667498776078531363964430057e-01 2.52768105365115680616128202018444e-03 4.21248012078165237688401160986995e-07
            9.86271235609057494997387038893066e-01 1.32686523127691687745643989160271e-02 5.10913101944779995640995989836242e-06
            9.66485543413008074686842974188039e-01 3.23912377866705583628892384240316e-02 1.93346400269997297600640950854256e-05
            9.38376179135220755966884098597802e-01 5.95585319956820832820731936863012e-02 4.75362452733895640547868988168290e-05
            9.02444200809420005171546108613256e-01 9.42862695937980027638403157652647e-02 9.28851795385339854254497238272847e-05
            8.59330681565975140756563632749021e-01 1.35954862668245995971361139709188e-01 1.56967865926077116073625306036377e-04
            8.09804937881823172496353890892351e-01 1.83820777965756965155463831251836e-01 2.39587381583485439524167093239271e-04
            7.54750738923003927638433196989354e-01 2.37029865363639291597763758545625e-01 3.38699704951483015948759947022495e-04
            6.95150519015145573753500229940983e-01 2.94632616288820481376120596905821e-01 4.50490674148233335993829484777962e-04
            6.32067840485172394160429121257039e-01 3.55601113127809731473405463475501e-01 5.69589407858223225819549462300984e-04
            5.66628412149233096073430715478025e-01 4.18847374583706633099922100882395e-01 6.89404051269860984607618092923076e-04
            5.00000000000000000000000000000000e-01 4.83242771706503981832270255836193e-01 8.02556808004345240310783982096154e-04
            4.33371587850766792904266822006321e-01 5.47638168829301608120374567079125e-01 9.01387940168347384130298483029264e-04
            3.67932159514827383794965953711653e-01 6.10884430285198565258042435743846e-01 9.78493283824150798105856452480111e-04
            3.04849480984854481757651001316844e-01 6.71852927124187426777268683508737e-01 1.02725720553608311502724603769821e-03
            2.45249261076996127872718034268473e-01 7.29455678049368838600230446900241e-01 1.04234300834397919897911233277910e-03
            1.90195062118176771992494877849822e-01 7.82664765447251276064832836709684e-01 1.02010558265665913189879088207590e-03
            1.40669318434024914754587598508806e-01 8.30530680744762106471057450107764e-01 9.58896401232484834123437344999275e-04
            9.75557991905800503396051226445707e-02 8.72199273819210030289639234979404e-01 8.59238428788214911140108664966419e-04
            6.16238208647792440331159014021978e-02 9.06927011417326012221451492223423e-01 7.238577482554715315787863