
function [xyw,pointset_stats]=set_papanicolopulos_A_I_standard(deg)

%--------------------------------------------------------------------
% CUBATURE RULE ON THE SIMPLEX [0 0; 0 1; 1 0] WITH ADE EQUAL TO deg.
% THE CUBATURE RULE IS DESCRIBED IN STANDARD FORM.
%--------------------------------------------------------------------
% INPUT:
% deg: algebraic degree of precision (ADE) of the rule.
%--------------------------------------------------------------------
% OUTPUT:
% xyw: MATRIX OF NODES xyw(:,1:2) AND WEIGHTS xyw(:,3).
%      REMARK: The sum of the weights makes 1/2.
%
% pointset_stats:
%    statistics of the pointset stored in a matrix. Each rows contains
%    degree, cardinality of the full pointset, moment error w.r.t. orthn.
%    Dubiner basis,number of points with negative weights, number of
%    points outside the domain.
%--------------------------------------------------------------------
% SOURCE:
% S-A. Papanicoloupolos
% "New fully symmetric and rotationally symmetric cubature rules on the
% triangle using minimal orthonormal bases".
% JCAM 294 (2016), 39-48.
% ** Using file fullsymmetry.txt from archives.
%--------------------------------------------------------------------
% | DEG | CARD | W- | PT.O|   M.E.   |Q.|
%--------------------------------------------------------------------
% |  15 |  48 |   3 |   0 | 5.41e-16 |NI|
% |  17 |  58 |   0 |   6 | 7.77e-16 |P0|
% |  21 |  87 |   0 |   0 | 5.24e-16 |PI|
% |  22 |  94 |   0 |   6 | 1.63e-15 |P0|
% |  23 | 102 |   0 |   0 | 9.72e-16 |PI|
% |  24 | 112 |   0 |   0 | 1.37e-15 |PI|
% |  25 | 118 |   0 |   6 | 6.75e-16 |P0|
%--------------------------------------------------------------------
% LEGEND:
%--------------------------------------------------------------------
% DEG  : algebraic degree of precision
% CARD : cardinality of the pointset
% W-   : negative weights
% PT.O : points outside the simplex [0 0; 1 0; 0 1].
% M.E. : maximum moment error in inf norm.
% Q.: quality of the rule.
%        PI: pos. weights, points in simplex
%        NI: some neg. weights, points in simplex
%        P0: pos. weights, some points out of the simplex
%        N0: some neg. weights, some points out of the simplex
%--------------------------------------------------------------------


switch deg
    
    case  15
        
        % ALG. DEG.:   15
        % PTS CARD.:   48
        % NEG. W.  :    3
        % OUT PTS. :    0
        % M.E.INF. : 5.4e-16
        
        xyw=[
            2.95932339143480549470410778667429e-01 2.95932339143480549470410778667429e-01 -4.37281629132241639323908088954340e-02
            2.95932339143480549470410778667429e-01 4.08135321713038901059178442665143e-01 -4.37281629132241639323908088954340e-02
            4.08135321713038901059178442665143e-01 2.95932339143480549470410778667429e-01 -4.37281629132241639323908088954340e-02
            1.10411372943988075889798494699789e-01 1.10411372943988075889798494699789e-01 5.91267986695783939743442658709682e-03
            1.10411372943988075889798494699789e-01 7.79177254112023875975978626229335e-01 5.91267986695783939743442658709682e-03
            7.79177254112023875975978626229335e-01 1.10411372943988075889798494699789e-01 5.91267986695783939743442658709682e-03
            5.55567775910977765252418691943603e-02 5.55567775910977765252418691943603e-02 6.81667990029549784175566529142998e-03
            5.55567775910977765252418691943603e-02 8.88886444817804433071728453796823e-01 6.81667990029549784175566529142998e-03
            8.88886444817804433071728453796823e-01 5.55567775910977765252418691943603e-02 6.81667990029549784175566529142998e-03
            4.90536861361350884358500934467884e-01 4.90536861361350884358500934467884e-01 9.07637490182124605231361869073226e-03
            4.90536861361350884358500934467884e-01 1.89262772772982312829981310642324e-02 9.07637490182124605231361869073226e-03
            1.89262772772982312829981310642324e-02 4.90536861361350884358500934467884e-01 9.07637490182124605231361869073226e-03
            2.32611403922656861942286354860698e-01 2.32611403922656861942286354860698e-01 3.00755390601345172496561275465865e-02
            2.32611403922656861942286354860698e-01 5.34777192154686220604276059020776e-01 3.00755390601345172496561275465865e-02
            5.34777192154686220604276059020776e-01 2.32611403922656861942286354860698e-01 3.00755390601345172496561275465865e-02
            3.70936506366430440717607552869595e-01 3.70936506366430440717607552869595e-01 6.10900602972157050207790973672672e-02
            3.70936506366430440717607552869595e-01 2.58126987267139118564784894260811e-01 6.10900602972157050207790973672672e-02
            2.58126987267139118564784894260811e-01 3.70936506366430440717607552869595e-01 6.10900602972157050207790973672672e-02
            1.77079308752543730827444523612257e-03 3.82312373532611438076145304876263e-02 1.28068599043408494660145624521874e-03
            1.77079308752543730827444523612257e-03 9.59997969559213348844650681712665e-01 1.28068599043408494660145624521874e-03
            3.82312373532611438076145304876263e-02 1.77079308752543730827444523612257e-03 1.28068599043408494660145624521874e-03
            3.82312373532611438076145304876263e-02 9.59997969559213348844650681712665e-01 1.28068599043408494660145624521874e-03
            9.59997969559213348844650681712665e-01 1.77079308752543730827444523612257e-03 1.28068599043408494660145624521874e-03
            9.59997969559213348844650681712665e-01 3.82312373532611438076145304876263e-02 1.28068599043408494660145624521874e-03
            1.74998973013415944077753039209711e-02 1.49819041618278969707489522988908e-01 6.09937568603339825418219177777246e-03
            1.74998973013415944077753039209711e-02 8.32681061080379425476394317229278e-01 6.09937568603339825418219177777246e-03
            1.49819041618278969707489522988908e-01 1.74998973013415944077753039209711e-02 6.09937568603339825418219177777246e-03
            1.49819041618278969707489522988908e-01 8.32681061080379425476394317229278e-01 6.09937568603339825418219177777246e-03
            8.32681061080379425476394317229278e-01 1.74998973013415944077753039209711e-02 6.09937568603339825418219177777246e-03
            8.32681061080379425476394317229278e-01 1.49819041618278969707489522988908e-01 6.09937568603339825418219177777246e-03
            1.84230652117384832711088904488861e-02 3.07452940137402042086733899850515e-01 8.21847168472195424571147270853544e-03
            1.84230652117384832711088904488861e-02 6.74123994650859481581051113607828e-01 8.21847168472195424571147270853544e-03
            3.07452940137402042086733899850515e-01 1.84230652117384832711088904488861e-02 8.21847168472195424571147270853544e-03
            3.07452940137402042086733899850515e-01 6.74123994650859481581051113607828e-01 8.21847168472195424571147270853544e-03
            6.74123994650859481581051113607828e-01 1.84230652117384832711088904488861e-02 8.21847168472195424571147270853544e-03
            6.74123994650859481581051113607828e-01 3.07452940137402042086733899850515e-01 8.21847168472195424571147270853544e-03
            9.12052725923131191176906895634602e-02 2.03858752868167986127545532326621e-01 1.41932760303792107614562212347664e-02
            9.12052725923131191176906895634602e-02 7.04935974539518839243612546852091e-01 1.41932760303792107614562212347664e-02
            2.03858752868167986127545532326621e-01 9.12052725923131191176906895634602e-02 1.41932760303792107614562212347664e-02
            2.03858752868167986127545532326621e-01 7.04935974539518839243612546852091e-01 1.41932760303792107614562212347664e-02
            7.04935974539518839243612546852091e-01 9.12052725923131191176906895634602e-02 1.41932760303792107614562212347664e-02
            7.04935974539518839243612546852091e-01 2.03858752868167986127545532326621e-01 1.41932760303792107614562212347664e-02
            9.75429348243705002241554780084698e-02 3.63255400132045880301490115016350e-01 1.89199383851643837539668879799137e-02
            9.75429348243705002241554780084698e-02 5.39201665043583577840990983531810e-01 1.89199383851643837539668879799137e-02
            3.63255400132045880301490115016350e-01 9.75429348243705002241554780084698e-02 1.89199383851643837539668879799137e-02
            3.63255400132045880301490115016350e-01 5.39201665043583577840990983531810e-01 1.89199383851643837539668879799137e-02
            5.39201665043583577840990983531810e-01 9.75429348243705002241554780084698e-02 1.89199383851643837539668879799137e-02
            5.39201665043583577840990983531810e-01 3.63255400132045880301490115016350e-01 1.89199383851643837539668879799137e-02
            ];
        
    case  17
        
        % ALG. DEG.:   17
        % PTS CARD.:   58
        % NEG. W.  :    0
        % OUT PTS. :    6
        % M.E.INF. : 7.8e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 2.11747601284573011559686506188882e-02
            1.49390638146109835243668939597228e-02 1.49390638146109835243668939597228e-02 1.45983451728896656759970618111311e-03
            1.49390638146109835243668939597228e-02 9.70121872370778071115182683570310e-01 1.45983451728896656759970618111311e-03
            9.70121872370778071115182683570310e-01 1.49390638146109835243668939597228e-02 1.45983451728896656759970618111311e-03
            4.95097032924142199661332597315777e-01 4.95097032924142199661332597315777e-01 4.66690967574416408492066210556004e-03
            4.95097032924142199661332597315777e-01 9.80593415171560067733480536844581e-03 4.66690967574416408492066210556004e-03
            9.80593415171560067733480536844581e-03 4.95097032924142199661332597315777e-01 4.66690967574416408492066210556004e-03
            8.01765199090131580605955718965561e-02 8.01765199090131580605955718965561e-02 7.78960500456638438998568929605426e-03
            8.01765199090131580605955718965561e-02 8.39646960181973711634384471835801e-01 7.78960500456638438998568929605426e-03
            8.39646960181973711634384471835801e-01 8.01765199090131580605955718965561e-02 7.78960500456638438998568929605426e-03
            4.66726138086600184085028786284965e-01 4.66726138086600184085028786284965e-01 1.29311919975470483029633683713655e-02
            4.66726138086600184085028786284965e-01 6.65477238267996318299424274300691e-02 1.29311919975470483029633683713655e-02
            6.65477238267996318299424274300691e-02 4.66726138086600184085028786284965e-01 1.29311919975470483029633683713655e-02
            1.59149357451283113285001036274480e-01 1.59149357451283113285001036274480e-01 1.35787567877466430754651582901715e-02
            1.59149357451283113285001036274480e-01 6.81701285097433773429997927451041e-01 1.35787567877466430754651582901715e-02
            6.81701285097433773429997927451041e-01 1.59149357451283113285001036274480e-01 1.35787567877466430754651582901715e-02
            4.09581021965064795420374821333098e-01 4.09581021965064795420374821333098e-01 1.91131032985946587576808752828583e-02
            4.09581021965064795420374821333098e-01 1.80837956069870409159250357333804e-01 1.91131032985946587576808752828583e-02
            1.80837956069870409159250357333804e-01 4.09581021965064795420374821333098e-01 1.91131032985946587576808752828583e-02
            2.44597319860783118938840630107734e-01 2.44597319860783118938840630107734e-01 1.93046958417219180481616547240264e-02
            2.44597319860783118938840630107734e-01 5.10805360278433706611167508526705e-01 1.93046958417219180481616547240264e-02
            5.10805360278433706611167508526705e-01 2.44597319860783118938840630107734e-01 1.93046958417219180481616547240264e-02
            -1.70992795947950831925155057433585e-02 3.04698462180811813215797201337409e-01 2.62998075919705924692604259718109e-04
            -1.70992795947950831925155057433585e-02 7.12400817413983311610081727849320e-01 2.62998075919705924692604259718109e-04
            3.04698462180811813215797201337409e-01 -1.70992795947950831925155057433585e-02 2.62998075919705924692604259718109e-04
            3.04698462180811813215797201337409e-01 7.12400817413983311610081727849320e-01 2.62998075919705924692604259718109e-04
            7.12400817413983311610081727849320e-01 -1.70992795947950831925155057433585e-02 2.62998075919705924692604259718109e-04
            7.12400817413983311610081727849320e-01 3.04698462180811813215797201337409e-01 2.62998075919705924692604259718109e-04
            1.56061494788847562181777206546940e-02 7.74677104516807435441450024882215e-02 3.41849025638204725985325183046371e-03
            1.56061494788847562181777206546940e-02 9.06926140069434505441847704787506e-01 3.41849025638204725985325183046371e-03
            7.74677104516807435441450024882215e-02 1.56061494788847562181777206546940e-02 3.41849025638204725985325183046371e-03
            7.74677104516807435441450024882215e-02 9.06926140069434505441847704787506e-01 3.41849025638204725985325183046371e-03
            9.06926140069434505441847704787506e-01 1.56061494788847562181777206546940e-02 3.41849025638204725985325183046371e-03
            9.06926140069434505441847704787506e-01 7.74677104516807435441450024882215e-02 3.41849025638204725985325183046371e-03
            1.24994725476382708556633005514414e-02 1.83007377411325222738014417700469e-01 3.87014909841642109136761185084197e-03
            1.24994725476382708556633005514414e-02 8.04493150041036497732704901864054e-01 3.87014909841642109136761185084197e-03
            1.83007377411325222738014417700469e-01 1.24994725476382708556633005514414e-02 3.87014909841642109136761185084197e-03
            1.83007377411325222738014417700469e-01 8.04493150041036497732704901864054e-01 3.87014909841642109136761185084197e-03
            8.04493150041036497732704901864054e-01 1.24994725476382708556633005514414e-02 3.87014909841642109136761185084197e-03
            8.04493150041036497732704901864054e-01 1.83007377411325222738014417700469e-01 3.87014909841642109136761185084197e-03
            2.48937969301927831955456582591069e-02 3.31361033435096818955400976847159e-01 8.65343584306812844963996411706830e-03
            2.48937969301927831955456582591069e-02 6.43745169634710401318500316847349e-01 8.65343584306812844963996411706830e-03
            3.31361033435096818955400976847159e-01 2.48937969301927831955456582591069e-02 8.65343584306812844963996411706830e-03
            3.31361033435096818955400976847159e-01 6.43745169634710401318500316847349e-01 8.65343584306812844963996411706830e-03
            6.43745169634710401318500316847349e-01 2.48937969301927831955456582591069e-02 8.65343584306812844963996411706830e-03
            6.43745169634710401318500316847349e-01 3.31361033435096818955400976847159e-01 8.65343584306812844963996411706830e-03
            6.09544889837634537577137905373093e-02 1.90839791501631328429766654153354e-01 8.81692420710726738886098274861070e-03
            6.09544889837634537577137905373093e-02 7.48205719514605238629201267031021e-01 8.81692420710726738886098274861070e-03
            1.90839791501631328429766654153354e-01 6.09544889837634537577137905373093e-02 8.81692420710726738886098274861070e-03
            1.90839791501631328429766654153354e-01 7.48205719514605238629201267031021e-01 8.81692420710726738886098274861070e-03
            7.48205719514605238629201267031021e-01 6.09544889837634537577137905373093e-02 8.81692420710726738886098274861070e-03
            7.48205719514605238629201267031021e-01 1.90839791501631328429766654153354e-01 8.81692420710726738886098274861070e-03
            1.10946632046319576536674844646768e-01 3.03713312450755168470095668453723e-01 1.53601606027585612518349122979089e-02
            1.10946632046319576536674844646768e-01 5.85340055502925227237653871270595e-01 1.53601606027585612518349122979089e-02
            3.03713312450755168470095668453723e-01 1.10946632046319576536674844646768e-01 1.53601606027585612518349122979089e-02
            3.03713312450755168470095668453723e-01 5.85340055502925227237653871270595e-01 1.53601606027585612518349122979089e-02
            5.85340055502925227237653871270595e-01 1.10946632046319576536674844646768e-01 1.53601606027585612518349122979089e-02
            5.85340055502925227237653871270595e-01 3.03713312450755168470095668453723e-01 1.53601606027585612518349122979089e-02
            ];
        
    case  21
        
        % ALG. DEG.:   21
        % PTS CARD.:   87
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.2e-16
        
        xyw=[
            1.08306479138961346542213703969537e-02 1.08306479138961346542213703969537e-02 7.66261070785149945791547843043645e-04
            1.08306479138961346542213703969537e-02 9.78338704172207695997087739669951e-01 7.66261070785149945791547843043645e-04
            9.78338704172207695997087739669951e-01 1.08306479138961346542213703969537e-02 7.66261070785149945791547843043645e-04
            4.98681169280067615456886187530472e-01 4.98681169280067615456886187530472e-01 1.47982104290044587033281509036442e-03
            4.98681169280067615456886187530472e-01 2.63766143986476908622762493905611e-03 1.47982104290044587033281509036442e-03
            2.63766143986476908622762493905611e-03 4.98681169280067615456886187530472e-01 1.47982104290044587033281509036442e-03
            5.32063025937022163192047230495518e-02 5.32063025937022163192047230495518e-02 3.55253597599161511158549764388681e-03
            5.32063025937022163192047230495518e-02 8.93587394812595525728227130457526e-01 3.55253597599161511158549764388681e-03
            8.93587394812595525728227130457526e-01 5.32063025937022163192047230495518e-02 3.55253597599161511158549764388681e-03
            4.84368806799071960345060006147833e-01 4.84368806799071960345060006147833e-01 6.21101272507559951790989671849275e-03
            4.84368806799071960345060006147833e-01 3.12623864018560793098799877043348e-02 6.21101272507559951790989671849275e-03
            3.12623864018560793098799877043348e-02 4.84368806799071960345060006147833e-01 6.21101272507559951790989671849275e-03
            1.17179324464095319102696635127359e-01 1.17179324464095319102696635127359e-01 6.57692654527430430738688116321100e-03
            1.17179324464095319102696635127359e-01 7.65641351071809417305757961003110e-01 6.57692654527430430738688116321100e-03
            7.65641351071809417305757961003110e-01 1.17179324464095319102696635127359e-01 6.57692654527430430738688116321100e-03
            1.86199874531600445548562561270955e-01 1.86199874531600445548562561270955e-01 8.05355996677970843722338400993976e-03
            1.86199874531600445548562561270955e-01 6.27600250936799053391723646200262e-01 8.05355996677970843722338400993976e-03
            6.27600250936799053391723646200262e-01 1.86199874531600445548562561270955e-01 8.05355996677970843722338400993976e-03
            2.37705046970502376080247586287442e-01 2.37705046970502376080247586287442e-01 9.77849916371077015897483875050966e-03
            2.37705046970502376080247586287442e-01 5.24589906058995247839504827425117e-01 9.77849916371077015897483875050966e-03
            5.24589906058995247839504827425117e-01 2.37705046970502376080247586287442e-01 9.77849916371077015897483875050966e-03
            4.51118096863331430323995618891786e-01 4.51118096863331430323995618891786e-01 1.05735729017055815615044878086337e-02
            4.51118096863331430323995618891786e-01 9.77638062733371393520087622164283e-02 1.05735729017055815615044878086337e-02
            9.77638062733371393520087622164283e-02 4.51118096863331430323995618891786e-01 1.05735729017055815615044878086337e-02
            2.98901616618379606382660540475626e-01 2.98901616618379606382660540475626e-01 1.07436248090477717148649716705222e-02
            2.98901616618379606382660540475626e-01 4.02196766763240787234678919048747e-01 1.07436248090477717148649716705222e-02
            4.02196766763240787234678919048747e-01 2.98901616618379606382660540475626e-01 1.07436248090477717148649716705222e-02
            1.01703755139702384929778489208729e-02 5.61060787404411287493388726943522e-02 1.62160801814857993830010229885374e-03
            1.01703755139702384929778489208729e-02 9.33723545745588578981255523103755e-01 1.62160801814857993830010229885374e-03
            5.61060787404411287493388726943522e-02 1.01703755139702384929778489208729e-02 1.62160801814857993830010229885374e-03
            5.61060787404411287493388726943522e-02 9.33723545745588578981255523103755e-01 1.62160801814857993830010229885374e-03
            9.33723545745588578981255523103755e-01 1.01703755139702384929778489208729e-02 1.62160801814857993830010229885374e-03
            9.33723545745588578981255523103755e-01 5.61060787404411287493388726943522e-02 1.62160801814857993830010229885374e-03
            7.40148001912942655328153662708246e-03 2.38060908939278198515054896233778e-01 2.22303341228364875412082390937485e-03
            7.40148001912942655328153662708246e-03 7.54537611041592337635108833637787e-01 2.22303341228364875412082390937485e-03
            2.38060908939278198515054896233778e-01 7.40148001912942655328153662708246e-03 2.22303341228364875412082390937485e-03
            2.38060908939278198515054896233778e-01 7.54537611041592337635108833637787e-01 2.22303341228364875412082390937485e-03
            7.54537611041592337635108833637787e-01 7.40148001912942655328153662708246e-03 2.22303341228364875412082390937485e-03
            7.54537611041592337635108833637787e-01 2.38060908939278198515054896233778e-01 2.22303341228364875412082390937485e-03
            9.48888021003903639472021325218520e-03 1.33714236127292523947929225869302e-01 2.22356850638449378421146462869729e-03
            9.48888021003903639472021325218520e-03 8.56796883662668484760160936275497e-01 2.22356850638449378421146462869729e-03
            1.33714236127292523947929225869302e-01 9.48888021003903639472021325218520e-03 2.22356850638449378421146462869729e-03
            1.33714236127292523947929225869302e-01 8.56796883662668484760160936275497e-01 2.22356850638449378421146462869729e-03
            8.56796883662668484760160936275497e-01 9.48888021003903639472021325218520e-03 2.22356850638449378421146462869729e-03
            8.56796883662668484760160936275497e-01 1.33714236127292523947929225869302e-01 2.22356850638449378421146462869729e-03
            1.07870852895329272236324413825059e-02 3.60878270361336406235608365022927e-01 3.59641504489720308448963415060007e-03
            1.07870852895329272236324413825059e-02 6.28334644349130666540759193594567e-01 3.59641504489720308448963415060007e-03
            3.60878270361336406235608365022927e-01 1.07870852895329272236324413825059e-02 3.59641504489720308448963415060007e-03
            3.60878270361336406235608365022927e-01 6.28334644349130666540759193594567e-01 3.59641504489720308448963415060007e-03
            6.28334644349130666540759193594567e-01 1.07870852895329272236324413825059e-02 3.59641504489720308448963415060007e-03
            6.28334644349130666540759193594567e-01 3.60878270361336406235608365022927e-01 3.59641504489720308448963415060007e-03
            4.88864101564119649112605259233533e-02 1.28560007074346699385003489624069e-01 4.84409138483868090957473739877059e-03
            4.88864101564119649112605259233533e-02 8.22553582769241398153781119617634e-01 4.84409138483868090957473739877059e-03
            1.28560007074346699385003489624069e-01 4.88864101564119649112605259233533e-02 4.84409138483868090957473739877059e-03
            1.28560007074346699385003489624069e-01 8.22553582769241398153781119617634e-01 4.84409138483868090957473739877059e-03
            8.22553582769241398153781119617634e-01 4.88864101564119649112605259233533e-02 4.84409138483868090957473739877059e-03
            8.22553582769241398153781119617634e-01 1.28560007074346699385003489624069e-01 4.84409138483868090957473739877059e-03
            4.07904784349213886396334771688998e-02 2.31929220198316188294995754404226e-01 5.38593362737300497544712385433741e-03
            4.07904784349213886396334771688998e-02 7.27280301366762471637628095777472e-01 5.38593362737300497544712385433741e-03
            2.31929220198316188294995754404226e-01 4.07904784349213886396334771688998e-02 5.38593362737300497544712385433741e-03
            2.31929220198316188294995754404226e-01 7.27280301366762471637628095777472e-01 5.38593362737300497544712385433741e-03
            7.27280301366762471637628095777472e-01 4.07904784349213886396334771688998e-02 5.38593362737300497544712385433741e-03
            7.27280301366762471637628095777472e-01 2.31929220198316188294995754404226e-01 5.38593362737300497544712385433741e-03
            1.02254765254272128593981960875681e-01 2.13182380246852126726153642266581e-01 7.68822439613710099937105013623295e-03
            1.02254765254272128593981960875681e-01 6.84562854498875772435440012486652e-01 7.68822439613710099937105013623295e-03
            2.13182380246852126726153642266581e-01 1.02254765254272128593981960875681e-01 7.68822439613710099937105013623295e-03
            2.13182380246852126726153642266581e-01 6.84562854498875772435440012486652e-01 7.68822439613710099937105013623295e-03
            6.84562854498875772435440012486652e-01 1.02254765254272128593981960875681e-01 7.68822439613710099937105013623295e-03
            6.84562854498875772435440012486652e-01 2.13182380246852126726153642266581e-01 7.68822439613710099937105013623295e-03
            5.61951037264934380455194684600428e-02 3.46153638930814133711777458302095e-01 7.70910081252550172248305671018898e-03
            5.61951037264934380455194684600428e-02 5.97651257342692421303809169330634e-01 7.70910081252550172248305671018898e-03
            3.46153638930814133711777458302095e-01 5.61951037264934380455194684600428e-02 7.70910081252550172248305671018898e-03
            3.46153638930814133711777458302095e-01 5.97651257342692421303809169330634e-01 7.70910081252550172248305671018898e-03
            5.97651257342692421303809169330634e-01 5.61951037264934380455194684600428e-02 7.70910081252550172248305671018898e-03
            5.97651257342692421303809169330634e-01 3.46153638930814133711777458302095e-01 7.70910081252550172248305671018898e-03
            1.93921114022049423386562239102204e-01 3.57417794125976018282386803548434e-01 9.36399704091174858722190776916250e-03
            1.93921114022049423386562239102204e-01 4.48661091851974613842202188607189e-01 9.36399704091174858722190776916250e-03
            3.57417794125976018282386803548434e-01 1.93921114022049423386562239102204e-01 9.36399704091174858722190776916250e-03
            3.57417794125976018282386803548434e-01 4.48661091851974613842202188607189e-01 9.36399704091174858722190776916250e-03
            4.48661091851974613842202188607189e-01 1.93921114022049423386562239102204e-01 9.36399704091174858722190776916250e-03
            4.48661091851974613842202188607189e-01 3.57417794125976018282386803548434e-01 9.36399704091174858722190776916250e-03
            1.31396937599549262776221780768537e-01 3.09641800485391083253006172526511e-01 9.80945398919793633274455402215608e-03
            1.31396937599549262776221780768537e-01 5.58961261915059681726347662333865e-01 9.80945398919793633274455402215608e-03
            3.09641800485391083253006172526511e-01 1.31396937599549262776221780768537e-01 9.80945398919793633274455402215608e-03
            3.09641800485391083253006172526511e-01 5.58961261915059681726347662333865e-01 9.80945398919793633274455402215608e-03
            5.58961261915059681726347662333865e-01 1.31396937599549262776221780768537e-01 9.80945398919793633274455402215608e-03
            5.58961261915059681726347662333865e-01 3.09641800485391083253006172526511e-01 9.80945398919793633274455402215608e-03
            ];
        
    case  22
        
        % ALG. DEG.:   22
        % PTS CARD.:   94
        % NEG. W.  :    0
        % OUT PTS. :    6
        % M.E.INF. : 1.6e-15
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 9.02303250275967097526752525027405e-03
            1.04964923185328299487517611510157e-02 1.04964923185328299487517611510157e-02 7.22027789765989887540986735814386e-04
            1.04964923185328299487517611510157e-02 9.79007015362934329694155621837126e-01 7.22027789765989887540986735814386e-04
            9.79007015362934329694155621837126e-01 1.04964923185328299487517611510157e-02 7.22027789765989887540986735814386e-04
            4.95533645188751648014147122012218e-01 4.95533645188751648014147122012218e-01 3.13323322477999065271969669765895e-03
            4.95533645188751648014147122012218e-01 8.93270962249670397170575597556308e-03 3.13323322477999065271969669765895e-03
            8.93270962249670397170575597556308e-03 4.95533645188751648014147122012218e-01 3.13323322477999065271969669765895e-03
            5.60496151328661179613632725704520e-02 5.60496151328661179613632725704520e-02 3.82587218006151140101134267013094e-03
            5.60496151328661179613632725704520e-02 8.87900769734267791832849070488010e-01 3.82587218006151140101134267013094e-03
            8.87900769734267791832849070488010e-01 5.60496151328661179613632725704520e-02 3.82587218006151140101134267013094e-03
            4.76665205864216678932621107378509e-01 4.76665205864216678932621107378509e-01 6.63992389205046631278639424067478e-03
            4.76665205864216678932621107378509e-01 4.66695882715666421347577852429822e-02 6.63992389205046631278639424067478e-03
            4.66695882715666421347577852429822e-02 4.76665205864216678932621107378509e-01 6.63992389205046631278639424067478e-03
            1.31716048354042047918710522935726e-01 1.31716048354042047918710522935726e-01 7.55506000191125473564035530671390e-03
            1.31716048354042047918710522935726e-01 7.36567903291915904162578954128549e-01 7.55506000191125473564035530671390e-03
            7.36567903291915904162578954128549e-01 1.31716048354042047918710522935726e-01 7.55506000191125473564035530671390e-03
            4.43302304754163378763820446692989e-01 4.43302304754163378763820446692989e-01 8.79594269443569748423428222849907e-03
            4.43302304754163378763820446692989e-01 1.13395390491673242472359106614022e-01 8.79594269443569748423428222849907e-03
            1.13395390491673242472359106614022e-01 4.43302304754163378763820446692989e-01 8.79594269443569748423428222849907e-03
            3.94784015644457919602672291148338e-01 3.94784015644457919602672291148338e-01 9.42185419696263230793231713278146e-03
            3.94784015644457919602672291148338e-01 2.10431968711084160794655417703325e-01 9.42185419696263230793231713278146e-03
            2.10431968711084160794655417703325e-01 3.94784015644457919602672291148338e-01 9.42185419696263230793231713278146e-03
            2.11601327317755272261479149165098e-01 2.11601327317755272261479149165098e-01 9.92411679974057950315291520837491e-03
            2.11601327317755272261479149165098e-01 5.76797345364489455477041701669805e-01 9.92411679974057950315291520837491e-03
            5.76797345364489455477041701669805e-01 2.11601327317755272261479149165098e-01 9.92411679974057950315291520837491e-03
            2.80558889938600508617838613645290e-01 2.80558889938600508617838613645290e-01 1.04335812393143937543138477508364e-02
            2.80558889938600508617838613645290e-01 4.38882220122798982764322772709420e-01 1.04335812393143937543138477508364e-02
            4.38882220122798982764322772709420e-01 2.80558889938600508617838613645290e-01 1.04335812393143937543138477508364e-02
            -3.65758201288990725252858737803763e-01 1.21775779289979721453640593153978e-01 1.20072435555068730056227794636732e-12
            -3.65758201288990725252858737803763e-01 1.24398242199901121196603526186664e+00 1.20072435555068730056227794636732e-12
            1.21775779289979721453640593153978e-01 -3.65758201288990725252858737803763e-01 1.20072435555068730056227794636732e-12
            1.21775779289979721453640593153978e-01 1.24398242199901121196603526186664e+00 1.20072435555068730056227794636732e-12
            1.24398242199901121196603526186664e+00 -3.65758201288990725252858737803763e-01 1.20072435555068730056227794636732e-12
            1.24398242199901121196603526186664e+00 1.21775779289979721453640593153978e-01 1.20072435555068730056227794636732e-12
            1.08660845048417047109534649962370e-02 5.46990068206437349274118275843648e-02 1.69584936297267289267987067091781e-03
            1.08660845048417047109534649962370e-02 9.34434908674514508319930428115185e-01 1.69584936297267289267987067091781e-03
            5.46990068206437349274118275843648e-02 1.08660845048417047109534649962370e-02 1.69584936297267289267987067091781e-03
            5.46990068206437349274118275843648e-02 9.34434908674514508319930428115185e-01 1.69584936297267289267987067091781e-03
            9.34434908674514508319930428115185e-01 1.08660845048417047109534649962370e-02 1.69584936297267289267987067091781e-03
            9.34434908674514508319930428115185e-01 5.46990068206437349274118275843648e-02 1.69584936297267289267987067091781e-03
            6.98641020724628195731620294850472e-03 3.60846088228242101170195610393421e-01 2.37133818093425473641500111909863e-03
            6.98641020724628195731620294850472e-03 6.32167501564511624678743828553706e-01 2.37133818093425473641500111909863e-03
            3.60846088228242101170195610393421e-01 6.98641020724628195731620294850472e-03 2.37133818093425473641500111909863e-03
            3.60846088228242101170195610393421e-01 6.32167501564511624678743828553706e-01 2.37133818093425473641500111909863e-03
            6.32167501564511624678743828553706e-01 6.98641020724628195731620294850472e-03 2.37133818093425473641500111909863e-03
            6.32167501564511624678743828553706e-01 3.60846088228242101170195610393421e-01 2.37133818093425473641500111909863e-03
            1.10884495183268746987259589786845e-02 1.31476833810282828229887286397570e-01 2.58599455351851605783286558448708e-03
            1.10884495183268746987259589786845e-02 8.57434716671390262376917235087603e-01 2.58599455351851605783286558448708e-03
            1.31476833810282828229887286397570e-01 1.10884495183268746987259589786845e-02 2.58599455351851605783286558448708e-03
            1.31476833810282828229887286397570e-01 8.57434716671390262376917235087603e-01 2.58599455351851605783286558448708e-03
            8.57434716671390262376917235087603e-01 1.10884495183268746987259589786845e-02 2.58599455351851605783286558448708e-03
            8.57434716671390262376917235087603e-01 1.31476833810282828229887286397570e-01 2.58599455351851605783286558448708e-03
            9.49913010597105804400097639472733e-03 2.35799841504738955100251018848212e-01 2.81405995560641327551953949637209e-03
            9.49913010597105804400097639472733e-03 7.54701028389289985121024528780254e-01 2.81405995560641327551953949637209e-03
            2.35799841504738955100251018848212e-01 9.49913010597105804400097639472733e-03 2.81405995560641327551953949637209e-03
            2.35799841504738955100251018848212e-01 7.54701028389289985121024528780254e-01 2.81405995560641327551953949637209e-03
            7.54701028389289985121024528780254e-01 9.49913010597105804400097639472733e-03 2.81405995560641327551953949637209e-03
            7.54701028389289985121024528780254e-01 2.35799841504738955100251018848212e-01 2.81405995560641327551953949637209e-03
            3.70466160480027797974322822938120e-02 3.53558464226123136953106040891726e-01 5.28697528670214224322254636945217e-03
            3.70466160480027797974322822938120e-02 6.09394919725874117943931196350604e-01 5.28697528670214224322254636945217e-03
            3.53558464226123136953106040891726e-01 3.70466160480027797974322822938120e-02 5.28697528670214224322254636945217e-03
            3.53558464226123136953106040891726e-01 6.09394919725874117943931196350604e-01 5.28697528670214224322254636945217e-03
            6.09394919725874117943931196350604e-01 3.70466160480027797974322822938120e-02 5.28697528670214224322254636945217e-03
            6.09394919725874117943931196350604e-01 3.53558464226123136953106040891726e-01 5.28697528670214224322254636945217e-03
            5.65981092525428552963973061196157e-02 1.32527176274670116518095142055245e-01 5.54330624111837831635707374289268e-03
            5.65981092525428552963973061196157e-02 8.10874714472787028185507551825140e-01 5.54330624111837831635707374289268e-03
            1.32527176274670116518095142055245e-01 5.65981092525428552963973061196157e-02 5.54330624111837831635707374289268e-03
            1.32527176274670116518095142055245e-01 8.10874714472787028185507551825140e-01 5.54330624111837831635707374289268e-03
            8.10874714472787028185507551825140e-01 5.65981092525428552963973061196157e-02 5.54330624111837831635707374289268e-03
            8.10874714472787028185507551825140e-01 1.32527176274670116518095142055245e-01 5.54330624111837831635707374289268e-03
            4.93070866574566307738436421459483e-02 2.33455411836817949833289276284631e-01 6.06725521080308909105394121752397e-03
            4.93070866574566307738436421459483e-02 7.17237501505725405515079273754964e-01 6.06725521080308909105394121752397e-03
            2.33455411836817949833289276284631e-01 4.93070866574566307738436421459483e-02 6.06725521080308909105394121752397e-03
            2.33455411836817949833289276284631e-01 7.17237501505725405515079273754964e-01 6.06725521080308909105394121752397e-03
            7.17237501505725405515079273754964e-01 4.93070866574566307738436421459483e-02 6.06725521080308909105394121752397e-03
            7.17237501505725405515079273754964e-01 2.33455411836817949833289276284631e-01 6.06725521080308909105394121752397e-03
            9.16967558296845758336601761584461e-02 3.40216421113571698686683930645813e-01 7.57718923793823228435284633519586e-03
            9.16967558296845758336601761584461e-02 5.68086823056743739357443701010197e-01 7.57718923793823228435284633519586e-03
            3.40216421113571698686683930645813e-01 9.16967558296845758336601761584461e-02 7.57718923793823228435284633519586e-03
            3.40216421113571698686683930645813e-01 5.68086823056743739357443701010197e-01 7.57718923793823228435284633519586e-03
            5.68086823056743739357443701010197e-01 9.16967558296845758336601761584461e-02 7.57718923793823228435284633519586e-03
            5.68086823056743739357443701010197e-01 3.40216421113571698686683930645813e-01 7.57718923793823228435284633519586e-03
            1.18093857071422919657877059762541e-01 2.28110979662494828268393121106783e-01 8.39933727273014726910638216850202e-03
            1.18093857071422919657877059762541e-01 6.53795163266082224318154203501763e-01 8.39933727273014726910638216850202e-03
            2.28110979662494828268393121106783e-01 1.18093857071422919657877059762541e-01 8.39933727273014726910638216850202e-03
            2.28110979662494828268393121106783e-01 6.53795163266082224318154203501763e-01 8.39933727273014726910638216850202e-03
            6.53795163266082224318154203501763e-01 1.18093857071422919657877059762541e-01 8.39933727273014726910638216850202e-03
            6.53795163266082224318154203501763e-01 2.28110979662494828268393121106783e-01 8.39933727273014726910638216850202e-03
            1.72252200982416225905069495638600e-01 3.19532176347331609012059061569744e-01 9.26238326983755758115002265640214e-03
            1.72252200982416225905069495638600e-01 5.08215622670252109571720211533830e-01 9.26238326983755758115002265640214e-03
            3.19532176347331609012059061569744e-01 1.72252200982416225905069495638600e-01 9.26238326983755758115002265640214e-03
            3.19532176347331609012059061569744e-01 5.08215622670252109571720211533830e-01 9.26238326983755758115002265640214e-03
            5.08215622670252109571720211533830e-01 1.72252200982416225905069495638600e-01 9.26238326983755758115002265640214e-03
            5.08215622670252109571720211533830e-01 3.19532176347331609012059061569744e-01 9.26238326983755758115002265640214e-03
            ];
        
    case  23
        
        % ALG. DEG.:   23
        % PTS CARD.:  102
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 9.7e-16
        
        xyw=[
            8.84309098564000847786381598325534e-03 8.84309098564000847786381598325534e-03 5.11692466965324378909418800986941e-04
            8.84309098564000847786381598325534e-03 9.82313818028719976105378464126261e-01 5.11692466965324378909418800986941e-04
            9.82313818028719976105378464126261e-01 8.84309098564000847786381598325534e-03 5.11692466965324378909418800986941e-04
            4.99042317995364748117737008215045e-01 4.99042317995364748117737008215045e-01 1.17476642805803763169869569082948e-03
            4.99042317995364748117737008215045e-01 1.91536400927050376452598356991075e-03 1.17476642805803763169869569082948e-03
            1.91536400927050376452598356991075e-03 4.99042317995364748117737008215045e-01 1.17476642805803763169869569082948e-03
            4.19184946319212034526735521922092e-02 4.19184946319212034526735521922092e-02 2.21136147415873333954561275049855e-03
            4.19184946319212034526735521922092e-02 9.16163010736157579216865087801125e-01 2.21136147415873333954561275049855e-03
            9.16163010736157579216865087801125e-01 4.19184946319212034526735521922092e-02 2.21136147415873333954561275049855e-03
            8.38439594007418936838504919251136e-02 8.38439594007418936838504919251136e-02 4.67969746729453729000569595086745e-03
            8.38439594007418936838504919251136e-02 8.32312081198516184876723400520859e-01 4.67969746729453729000569595086745e-03
            8.32312081198516184876723400520859e-01 8.38439594007418936838504919251136e-02 4.67969746729453729000569595086745e-03
            3.51566022256878052409945212275488e-01 3.51566022256878052409945212275488e-01 5.61165245397776072527884494434147e-03
            3.51566022256878052409945212275488e-01 2.96867955486243895180109575449023e-01 5.61165245397776072527884494434147e-03
            2.96867955486243895180109575449023e-01 3.51566022256878052409945212275488e-01 5.61165245397776072527884494434147e-03
            4.81422500082097581941553698925418e-01 4.81422500082097581941553698925418e-01 5.72769693035933471353393642289120e-03
            4.81422500082097581941553698925418e-01 3.71549998358048361168926021491643e-02 5.72769693035933471353393642289120e-03
            3.71549998358048361168926021491643e-02 4.81422500082097581941553698925418e-01 5.72769693035933471353393642289120e-03
            1.36036839176656848549029632522434e-01 1.36036839176656848549029632522434e-01 7.42576409513521877581165853143830e-03
            1.36036839176656848549029632522434e-01 7.27926321646686247390789503697306e-01 7.42576409513521877581165853143830e-03
            7.27926321646686247390789503697306e-01 1.36036839176656848549029632522434e-01 7.42576409513521877581165853143830e-03
            4.45886210232929069974261437891982e-01 4.45886210232929069974261437891982e-01 9.36699548876152032761410737293772e-03
            4.45886210232929069974261437891982e-01 1.08227579534141860051477124216035e-01 9.36699548876152032761410737293772e-03
            1.08227579534141860051477124216035e-01 4.45886210232929069974261437891982e-01 9.36699548876152032761410737293772e-03
            1.98337919690773967218433426751290e-01 1.98337919690773967218433426751290e-01 1.00477244490875094323767413584392e-02
            1.98337919690773967218433426751290e-01 6.03324160618452065563133146497421e-01 1.00477244490875094323767413584392e-02
            6.03324160618452065563133146497421e-01 1.98337919690773967218433426751290e-01 1.00477244490875094323767413584392e-02
            3.97648733525920727860381020946079e-01 3.97648733525920727860381020946079e-01 1.09347608012315168635186779511059e-02
            3.97648733525920727860381020946079e-01 2.04702532948158544279237958107842e-01 1.09347608012315168635186779511059e-02
            2.04702532948158544279237958107842e-01 3.97648733525920727860381020946079e-01 1.09347608012315168635186779511059e-02
            2.15375088982479138965953069373427e-03 1.11478221644365146913990827215457e-01 6.95187069452931683845309240865618e-04
            2.15375088982479138965953069373427e-03 8.86368027465810004450474934856175e-01 6.95187069452931683845309240865618e-04
            1.11478221644365146913990827215457e-01 2.15375088982479138965953069373427e-03 6.95187069452931683845309240865618e-04
            1.11478221644365146913990827215457e-01 8.86368027465810004450474934856175e-01 6.95187069452931683845309240865618e-04
            8.86368027465810004450474934856175e-01 2.15375088982479138965953069373427e-03 6.95187069452931683845309240865618e-04
            8.86368027465810004450474934856175e-01 1.11478221644365146913990827215457e-01 6.95187069452931683845309240865618e-04
            7.97567048343379887687110141314406e-03 4.59987619119273868917296965719288e-02 1.04680658585011533354136581408511e-03
            7.97567048343379887687110141314406e-03 9.46025567604638739638289735012222e-01 1.04680658585011533354136581408511e-03
            4.59987619119273868917296965719288e-02 7.97567048343379887687110141314406e-03 1.04680658585011533354136581408511e-03
            4.59987619119273868917296965719288e-02 9.46025567604638739638289735012222e-01 1.04680658585011533354136581408511e-03
            9.46025567604638739638289735012222e-01 7.97567048343379887687110141314406e-03 1.04680658585011533354136581408511e-03
            9.46025567604638739638289735012222e-01 4.59987619119273868917296965719288e-02 1.04680658585011533354136581408511e-03
            2.22521306870228970792302547465624e-03 2.99917045006213633939751161960885e-01 1.13136394558029291547429640019118e-03
            2.22521306870228970792302547465624e-03 6.97857741925084096301645786297740e-01 1.13136394558029291547429640019118e-03
            2.99917045006213633939751161960885e-01 2.22521306870228970792302547465624e-03 1.13136394558029291547429640019118e-03
            2.99917045006213633939751161960885e-01 6.97857741925084096301645786297740e-01 1.13136394558029291547429640019118e-03
            6.97857741925084096301645786297740e-01 2.22521306870228970792302547465624e-03 1.13136394558029291547429640019118e-03
            6.97857741925084096301645786297740e-01 2.99917045006213633939751161960885e-01 1.13136394558029291547429640019118e-03
            1.05679902342133303499993601803908e-02 1.94506172551481726351596535096178e-01 2.54943803926591322853156817984654e-03
            1.05679902342133303499993601803908e-02 7.94925837214304986666491004143609e-01 2.54943803926591322853156817984654e-03
            1.94506172551481726351596535096178e-01 1.05679902342133303499993601803908e-02 2.54943803926591322853156817984654e-03
            1.94506172551481726351596535096178e-01 7.94925837214304986666491004143609e-01 2.54943803926591322853156817984654e-03
            7.94925837214304986666491004143609e-01 1.05679902342133303499993601803908e-02 2.54943803926591322853156817984654e-03
            7.94925837214304986666491004143609e-01 1.94506172551481726351596535096178e-01 2.54943803926591322853156817984654e-03
            2.63901579600523857049854115075505e-02 1.04876013751831961839933171631856e-01 2.90139624317081908089210884327258e-03
            2.63901579600523857049854115075505e-02 8.68733828288115628168952753185295e-01 2.90139624317081908089210884327258e-03
            1.04876013751831961839933171631856e-01 2.63901579600523857049854115075505e-02 2.90139624317081908089210884327258e-03
            1.04876013751831961839933171631856e-01 8.68733828288115628168952753185295e-01 2.90139624317081908089210884327258e-03
            8.68733828288115628168952753185295e-01 2.63901579600523857049854115075505e-02 2.90139624317081908089210884327258e-03
            8.68733828288115628168952753185295e-01 1.04876013751831961839933171631856e-01 2.90139624317081908089210884327258e-03
            1.37266147889163348666530239938766e-02 3.93592741558122238831884942555916e-01 3.49523706008161762909258740705809e-03
            1.37266147889163348666530239938766e-02 5.92680643652961469669548932870384e-01 3.49523706008161762909258740705809e-03
            3.93592741558122238831884942555916e-01 1.37266147889163348666530239938766e-02 3.49523706008161762909258740705809e-03
            3.93592741558122238831884942555916e-01 5.92680643652961469669548932870384e-01 3.49523706008161762909258740705809e-03
            5.92680643652961469669548932870384e-01 1.37266147889163348666530239938766e-02 3.49523706008161762909258740705809e-03
            5.92680643652961469669548932870384e-01 3.93592741558122238831884942555916e-01 3.49523706008161762909258740705809e-03
            3.29316707171191241920737979853584e-02 2.77450101542849103530841148312902e-01 5.05272370634275019840586651298509e-03
            3.29316707171191241920737979853584e-02 6.89618227740031786154872861516196e-01 5.05272370634275019840586651298509e-03
            2.77450101542849103530841148312902e-01 3.29316707171191241920737979853584e-02 5.05272370634275019840586651298509e-03
            2.77450101542849103530841148312902e-01 6.89618227740031786154872861516196e-01 5.05272370634275019840586651298509e-03
            6.89618227740031786154872861516196e-01 3.29316707171191241920737979853584e-02 5.05272370634275019840586651298509e-03
            6.89618227740031786154872861516196e-01 2.77450101542849103530841148312902e-01 5.05272370634275019840586651298509e-03
            5.64583692867880043708694870474574e-02 1.70419309607223817826238132511207e-01 5.47034341005552329773031772219838e-03
            5.64583692867880043708694870474574e-02 7.73122321105988108413953341369051e-01 5.47034341005552329773031772219838e-03
            1.70419309607223817826238132511207e-01 5.64583692867880043708694870474574e-02 5.47034341005552329773031772219838e-03
            1.70419309607223817826238132511207e-01 7.73122321105988108413953341369051e-01 5.47034341005552329773031772219838e-03
            7.73122321105988108413953341369051e-01 5.64583692867880043708694870474574e-02 5.47034341005552329773031772219838e-03
            7.73122321105988108413953341369051e-01 1.70419309607223817826238132511207e-01 5.47034341005552329773031772219838e-03
            2.49058699506046365979017309655319e-01 2.80484645940626220461666662231437e-01 6.34887373360007445377561907662312e-03
            2.49058699506046365979017309655319e-01 4.70456654553327358048164796855417e-01 6.34887373360007445377561907662312e-03
            2.80484645940626220461666662231437e-01 2.49058699506046365979017309655319e-01 6.34887373360007445377561907662312e-03
            2.80484645940626220461666662231437e-01 4.70456654553327358048164796855417e-01 6.34887373360007445377561907662312e-03
            4.70456654553327358048164796855417e-01 2.49058699506046365979017309655319e-01 6.34887373360007445377561907662312e-03
            4.70456654553327358048164796855417e-01 2.80484645940626220461666662231437e-01 6.34887373360007445377561907662312e-03
            6.71002288455987278004144513943174e-02 3.59406843852176305365020425597322e-01 7.46845260736796971029471947645106e-03
            6.71002288455987278004144513943174e-02 5.73492927302224897445626083936077e-01 7.46845260736796971029471947645106e-03
            3.59406843852176305365020425597322e-01 6.71002288455987278004144513943174e-02 7.46845260736796971029471947645106e-03
            3.59406843852176305365020425597322e-01 5.73492927302224897445626083936077e-01 7.46845260736796971029471947645106e-03
            5.73492927302224897445626083936077e-01 6.71002288455987278004144513943174e-02 7.46845260736796971029471947645106e-03
            5.73492927302224897445626083936077e-01 3.59406843852176305365020425597322e-01 7.46845260736796971029471947645106e-03
            1.00958484047456359400385395019839e-01 2.39678973324078781192980613923282e-01 8.12297244448157741703031575752902e-03
            1.00958484047456359400385395019839e-01 6.59362542628464831651058375427965e-01 8.12297244448157741703031575752902e-03
            2.39678973324078781192980613923282e-01 1.00958484047456359400385395019839e-01 8.12297244448157741703031575752902e-03
            2.39678973324078781192980613923282e-01 6.59362542628464831651058375427965e-01 8.12297244448157741703031575752902e-03
            6.59362542628464831651058375427965e-01 1.00958484047456359400385395019839e-01 8.12297244448157741703031575752902e-03
            6.59362542628464831651058375427965e-01 2.39678973324078781192980613923282e-01 8.12297244448157741703031575752902e-03
            1.52081403116914759543476520775585e-01 3.16605382389148204413942266910453e-01 1.02044824605690403335200855394760e-02
            1.52081403116914759543476520775585e-01 5.31313214493936980531429981056135e-01 1.02044824605690403335200855394760e-02
            3.16605382389148204413942266910453e-01 1.52081403116914759543476520775585e-01 1.02044824605690403335200855394760e-02
            3.16605382389148204413942266910453e-01 5.31313214493936980531429981056135e-01 1.02044824605690403335200855394760e-02
            5.31313214493936980531429981056135e-01 1.52081403116914759543476520775585e-01 1.02044824605690403335200855394760e-02
            5.31313214493936980531429981056135e-01 3.16605382389148204413942266910453e-01 1.02044824605690403335200855394760e-02
            ];
        
    case  24
        
        % ALG. DEG.:   24
        % PTS CARD.:  112
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.4e-15
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 1.14689717404837434461262191121023e-02
            7.64481587209140354965963837230447e-03 7.64481587209140354965963837230447e-03 3.83688730557907005727646954085230e-04
            7.64481587209140354965963837230447e-03 9.84710368255817147797870347858407e-01 3.83688730557907005727646954085230e-04
            9.84710368255817147797870347858407e-01 7.64481587209140354965963837230447e-03 3.83688730557907005727646954085230e-04
            4.95123266934894168400660419138148e-01 4.95123266934894168400660419138148e-01 2.76178922237584837523494130095969e-03
            4.95123266934894168400660419138148e-01 9.75346613021166319867916172370315e-03 2.76178922237584837523494130095969e-03
            9.75346613021166319867916172370315e-03 4.95123266934894168400660419138148e-01 2.76178922237584837523494130095969e-03
            8.53357147077405536750305259374727e-02 8.53357147077405536750305259374727e-02 4.19249765137815443016844341173055e-03
            8.53357147077405536750305259374727e-02 8.29328570584518920405514563753968e-01 4.19249765137815443016844341173055e-03
            8.29328570584518920405514563753968e-01 8.53357147077405536750305259374727e-02 4.19249765137815443016844341173055e-03
            4.75404357579430003610809762903955e-01 4.75404357579430003610809762903955e-01 6.10869008900117443916499837541778e-03
            4.75404357579430003610809762903955e-01 4.91912848411399927783804741920903e-02 6.10869008900117443916499837541778e-03
            4.91912848411399927783804741920903e-02 4.75404357579430003610809762903955e-01 6.10869008900117443916499837541778e-03
            1.43613126002549662540985764280776e-01 1.43613126002549662540985764280776e-01 6.53519530280588945708730719275081e-03
            1.43613126002549662540985764280776e-01 7.12773747994900674918028471438447e-01 6.53519530280588945708730719275081e-03
            7.12773747994900674918028471438447e-01 1.43613126002549662540985764280776e-01 6.53519530280588945708730719275081e-03
            4.40001375500573721488706269155955e-01 4.40001375500573721488706269155955e-01 8.94061387921478781248829648120591e-03
            4.40001375500573721488706269155955e-01 1.19997248998852557022587461688090e-01 8.94061387921478781248829648120591e-03
            1.19997248998852557022587461688090e-01 4.40001375500573721488706269155955e-01 8.94061387921478781248829648120591e-03
            2.04386847871303928281250250620360e-01 2.04386847871303928281250250620360e-01 9.11087206954865405639232278645068e-03
            2.04386847871303928281250250620360e-01 5.91226304257392198948650730017107e-01 9.11087206954865405639232278645068e-03
            5.91226304257392198948650730017107e-01 2.04386847871303928281250250620360e-01 9.11087206954865405639232278645068e-03
            3.91555207032681984458122315118089e-01 3.91555207032681984458122315118089e-01 1.08207486087838061772359310452885e-02
            3.91555207032681984458122315118089e-01 2.16889585934636031083755369763821e-01 1.08207486087838061772359310452885e-02
            2.16889585934636031083755369763821e-01 3.91555207032681984458122315118089e-01 1.08207486087838061772359310452885e-02
            2.69360691849568234523815135617042e-01 2.69360691849568234523815135617042e-01 1.08441564634612994805218377791789e-02
            2.69360691849568234523815135617042e-01 4.61278616300863530952369728765916e-01 1.08441564634612994805218377791789e-02
            4.61278616300863530952369728765916e-01 2.69360691849568234523815135617042e-01 1.08441564634612994805218377791789e-02
            1.28915765322182144549112425835347e-03 3.89300321173704000354831578079029e-01 8.73092291330860594510132610679420e-04
            1.28915765322182144549112425835347e-03 6.09410521173074259948521103069652e-01 8.73092291330860594510132610679420e-04
            3.89300321173704000354831578079029e-01 1.28915765322182144549112425835347e-03 8.73092291330860594510132610679420e-04
            3.89300321173704000354831578079029e-01 6.09410521173074259948521103069652e-01 8.73092291330860594510132610679420e-04
            6.09410521173074259948521103069652e-01 1.28915765322182144549112425835347e-03 8.73092291330860594510132610679420e-04
            6.09410521173074259948521103069652e-01 3.89300321173704000354831578079029e-01 8.73092291330860594510132610679420e-04
            8.04615544134750990123983171997679e-03 3.99455189985108982675399147410644e-02 9.19285801871393332997117742166893e-04
            8.04615544134750990123983171997679e-03 9.52008325560141699384075764100999e-01 9.19285801871393332997117742166893e-04
            3.99455189985108982675399147410644e-02 8.04615544134750990123983171997679e-03 9.19285801871393332997117742166893e-04
            3.99455189985108982675399147410644e-02 9.52008325560141699384075764100999e-01 9.19285801871393332997117742166893e-04
            9.52008325560141699384075764100999e-01 8.04615544134750990123983171997679e-03 9.19285801871393332997117742166893e-04
            9.52008325560141699384075764100999e-01 3.99455189985108982675399147410644e-02 9.19285801871393332997117742166893e-04
            3.97463904079080419112779409829272e-03 1.77567553518447190930018564358761e-01 9.99468316681064285547653192054440e-04
            3.97463904079080419112779409829272e-03 8.18457807440762019624003187345807e-01 9.99468316681064285547653192054440e-04
            1.77567553518447190930018564358761e-01 3.97463904079080419112779409829272e-03 9.99468316681064285547653192054440e-04
            1.77567553518447190930018564358761e-01 8.18457807440762019624003187345807e-01 9.99468316681064285547653192054440e-04
            8.18457807440762019624003187345807e-01 3.97463904079080419112779409829272e-03 9.99468316681064285547653192054440e-04
            8.18457807440762019624003187345807e-01 1.77567553518447190930018564358761e-01 9.99468316681064285547653192054440e-04
            5.65339683877998013666887899830726e-03 9.73043731633908004496902321989182e-02 1.03184808566795377382518950071244e-03
            5.65339683877998013666887899830726e-03 8.97042229997829232424066958628828e-01 1.03184808566795377382518950071244e-03
            9.73043731633908004496902321989182e-02 5.65339683877998013666887899830726e-03 1.03184808566795377382518950071244e-03
            9.73043731633908004496902321989182e-02 8.97042229997829232424066958628828e-01 1.03184808566795377382518950071244e-03
            8.97042229997829232424066958628828e-01 5.65339683877998013666887899830726e-03 1.03184808566795377382518950071244e-03
            8.97042229997829232424066958628828e-01 9.73043731633908004496902321989182e-02 1.03184808566795377382518950071244e-03
            3.65416463470007168989894807964447e-02 4.73791364695558270270225875719916e-02 1.14769754357920772749879212426549e-03
            3.65416463470007168989894807964447e-02 9.16079217183443428318412316002650e-01 1.14769754357920772749879212426549e-03
            4.73791364695558270270225875719916e-02 3.65416463470007168989894807964447e-02 1.14769754357920772749879212426549e-03
            4.73791364695558270270225875719916e-02 9.16079217183443428318412316002650e-01 1.14769754357920772749879212426549e-03
            9.16079217183443428318412316002650e-01 3.65416463470007168989894807964447e-02 1.14769754357920772749879212426549e-03
            9.16079217183443428318412316002650e-01 4.73791364695558270270225875719916e-02 1.14769754357920772749879212426549e-03
            8.47203125278924405738845138102988e-03 2.76542334906226316970645484616398e-01 2.28976164445411547315734068774873e-03
            8.47203125278924405738845138102988e-03 7.14985633840984347031621837231796e-01 2.28976164445411547315734068774873e-03
            2.76542334906226316970645484616398e-01 8.47203125278924405738845138102988e-03 2.28976164445411547315734068774873e-03
            2.76542334906226316970645484616398e-01 7.14985633840984347031621837231796e-01 2.28976164445411547315734068774873e-03
            7.14985633840984347031621837231796e-01 8.47203125278924405738845138102988e-03 2.28976164445411547315734068774873e-03
            7.14985633840984347031621837231796e-01 2.76542334906226316970645484616398e-01 2.28976164445411547315734068774873e-03
            3.18687855584848661605334996238525e-02 1.01860048008413445197462010582967e-01 2.64661504954770161052168830906339e-03
            3.18687855584848661605334996238525e-02 8.66271166433101758030943528865464e-01 2.64661504954770161052168830906339e-03
            1.01860048008413445197462010582967e-01 3.18687855584848661605334996238525e-02 2.64661504954770161052168830906339e-03
            1.01860048008413445197462010582967e-01 8.66271166433101758030943528865464e-01 2.64661504954770161052168830906339e-03
            8.66271166433101758030943528865464e-01 3.18687855584848661605334996238525e-02 2.64661504954770161052168830906339e-03
            8.66271166433101758030943528865464e-01 1.01860048008413445197462010582967e-01 2.64661504954770161052168830906339e-03
            2.50744857569593221358061896353320e-02 1.79900411429897849613368521204393e-01 2.78937349799544039713472542985073e-03
            2.50744857569593221358061896353320e-02 7.95025102813142758861886250087991e-01 2.78937349799544039713472542985073e-03
            1.79900411429897849613368521204393e-01 2.50744857569593221358061896353320e-02 2.78937349799544039713472542985073e-03
            1.79900411429897849613368521204393e-01 7.95025102813142758861886250087991e-01 2.78937349799544039713472542985073e-03
            7.95025102813142758861886250087991e-01 2.50744857569593221358061896353320e-02 2.78937349799544039713472542985073e-03
            7.95025102813142758861886250087991e-01 1.79900411429897849613368521204393e-01 2.78937349799544039713472542985073e-03
            2.49546492072369834047673720078819e-02 3.74761488291451505538987021282082e-01 4.30966974362831985873034668088621e-03
            2.49546492072369834047673720078819e-02 6.00283862501311493709010846941965e-01 4.30966974362831985873034668088621e-03
            3.74761488291451505538987021282082e-01 2.49546492072369834047673720078819e-02 4.30966974362831985873034668088621e-03
            3.74761488291451505538987021282082e-01 6.00283862501311493709010846941965e-01 4.30966974362831985873034668088621e-03
            6.00283862501311493709010846941965e-01 2.49546492072369834047673720078819e-02 4.30966974362831985873034668088621e-03
            6.00283862501311493709010846941965e-01 3.74761488291451505538987021282082e-01 4.30966974362831985873034668088621e-03
            7.18069274899266751788218243746087e-02 1.62853713656418902511191504345334e-01 4.99617699612353409477361054769062e-03
            7.18069274899266751788218243746087e-02 7.65339358853654450065562286908971e-01 4.99617699612353409477361054769062e-03
            1.62853713656418902511191504345334e-01 7.18069274899266751788218243746087e-02 4.99617699612353409477361054769062e-03
            1.62853713656418902511191504345334e-01 7.65339358853654450065562286908971e-01 4.99617699612353409477361054769062e-03
            7.65339358853654450065562286908971e-01 7.18069274899266751788218243746087e-02 4.99617699612353409477361054769062e-03
            7.65339358853654450065562286908971e-01 1.62853713656418902511191504345334e-01 4.99617699612353409477361054769062e-03
            4.64953683252256055880025087390095e-02 2.62999946925326910651676826091716e-01 5.15280810384065313112245121374144e-03
            4.64953683252256055880025087390095e-02 6.90504684749447483760320665169274e-01 5.15280810384065313112245121374144e-03
            2.62999946925326910651676826091716e-01 4.64953683252256055880025087390095e-02 5.15280810384065313112245121374144e-03
            2.62999946925326910651676826091716e-01 6.90504684749447483760320665169274e-01 5.15280810384065313112245121374144e-03
            6.90504684749447483760320665169274e-01 4.64953683252256055880025087390095e-02 5.15280810384065313112245121374144e-03
            6.90504684749447483760320665169274e-01 2.62999946925326910651676826091716e-01 5.15280810384065313112245121374144e-03
            7.94456583289277762371938251817483e-02 3.53874622686032425100677301088581e-01 7.30227306312512206704079531505158e-03
            7.94456583289277762371938251817483e-02 5.66679718985039770906553258100757e-01 7.30227306312512206704079531505158e-03
            3.53874622686032425100677301088581e-01 7.94456583289277762371938251817483e-02 7.30227306312512206704079531505158e-03
            3.53874622686032425100677301088581e-01 5.66679718985039770906553258100757e-01 7.30227306312512206704079531505158e-03
            5.66679718985039770906553258100757e-01 7.94456583289277762371938251817483e-02 7.30227306312512206704079531505158e-03
            5.66679718985039770906553258100757e-01 3.53874622686032425100677301088581e-01 7.30227306312512206704079531505158e-03
            1.12104968398914525695708732655476e-01 2.40452720011096882091194970598735e-01 7.49925033812522991161220176081770e-03
            1.12104968398914525695708732655476e-01 6.47442311589988661602035335818073e-01 7.49925033812522991161220176081770e-03
            2.40452720011096882091194970598735e-01 1.12104968398914525695708732655476e-01 7.49925033812522991161220176081770e-03
            2.40452720011096882091194970598735e-01 6.47442311589988661602035335818073e-01 7.49925033812522991161220176081770e-03
            6.47442311589988661602035335818073e-01 1.12104968398914525695708732655476e-01 7.49925033812522991161220176081770e-03
            6.47442311589988661602035335818073e-01 2.40452720011096882091194970598735e-01 7.49925033812522991161220176081770e-03
            1.62887044320893126236882153534680e-01 3.16328836361366161344932379506645e-01 9.61539155871833813027382831251089e-03
            1.62887044320893126236882153534680e-01 5.20784119317740712418185466958676e-01 9.61539155871833813027382831251089e-03
            3.16328836361366161344932379506645e-01 1.62887044320893126236882153534680e-01 9.61539155871833813027382831251089e-03
            3.16328836361366161344932379506645e-01 5.20784119317740712418185466958676e-01 9.61539155871833813027382831251089e-03
            5.20784119317740712418185466958676e-01 1.62887044320893126236882153534680e-01 9.61539155871833813027382831251089e-03
            5.20784119317740712418185466958676e-01 3.16328836361366161344932379506645e-01 9.61539155871833813027382831251089e-03
            ];
        
    case  25
        
        % ALG. DEG.:   25
        % PTS CARD.:  118
        % NEG. W.  :    0
        % OUT PTS. :    6
        % M.E.INF. : 6.7e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 9.86507278208535384744770624365628e-03
            7.80221999790021124299155275139128e-03 7.80221999790021124299155275139128e-03 3.98924890045245612250207623006304e-04
            7.80221999790021124299155275139128e-03 9.84395560004199587922357750358060e-01 3.98924890045245612250207623006304e-04
            9.84395560004199587922357750358060e-01 7.80221999790021124299155275139128e-03 3.98924890045245612250207623006304e-04
            4.99200051265863864458083298814017e-01 4.99200051265863864458083298814017e-01 8.27027877955022845332666570072888e-04
            4.99200051265863864458083298814017e-01 1.59989746827227108383340237196535e-03 8.27027877955022845332666570072888e-04
            1.59989746827227108383340237196535e-03 4.99200051265863864458083298814017e-01 8.27027877955022845332666570072888e-04
            3.97459760601686362679885178295081e-02 3.97459760601686362679885178295081e-02 1.97415339077962890060313760898225e-03
            3.97459760601686362679885178295081e-02 9.20508047879662782975174195598811e-01 1.97415339077962890060313760898225e-03
            9.20508047879662782975174195598811e-01 3.97459760601686362679885178295081e-02 1.97415339077962890060313760898225e-03
            4.89484252465692326605761763858027e-01 4.89484252465692326605761763858027e-01 3.65296522097823764957391645680218e-03
            4.89484252465692326605761763858027e-01 2.10314950686153467884764722839464e-02 3.65296522097823764957391645680218e-03
            2.10314950686153467884764722839464e-02 4.89484252465692326605761763858027e-01 3.65296522097823764957391645680218e-03
            9.06944179019588681001096119871363e-02 9.06944179019588681001096119871363e-02 4.06392793982066589941748446790371e-03
            9.06944179019588681001096119871363e-02 8.18611164196082263799780776025727e-01 4.06392793982066589941748446790371e-03
            8.18611164196082263799780776025727e-01 9.06944179019588681001096119871363e-02 4.06392793982066589941748446790371e-03
            1.51224281357815043547176969696011e-01 1.51224281357815043547176969696011e-01 5.98518584278785129720867885794178e-03
            1.51224281357815043547176969696011e-01 6.97551437284369857394494829350151e-01 5.98518584278785129720867885794178e-03
            6.97551437284369857394494829350151e-01 1.51224281357815043547176969696011e-01 5.98518584278785129720867885794178e-03
            4.67213065208418099061304928909522e-01 4.67213065208418099061304928909522e-01 6.14988281310190846445928869457020e-03
            4.67213065208418099061304928909522e-01 6.55738695831638018773901421809569e-02 6.14988281310190846445928869457020e-03
            6.55738695831638018773901421809569e-02 4.67213065208418099061304928909522e-01 6.14988281310190846445928869457020e-03
            2.11913294950495378898835951986257e-01 2.11913294950495378898835951986257e-01 8.05635958802625694319399229925693e-03
            2.11913294950495378898835951986257e-01 5.76173410099009242202328096027486e-01 8.05635958802625694319399229925693e-03
            5.76173410099009242202328096027486e-01 2.11913294950495378898835951986257e-01 8.05635958802625694319399229925693e-03
            4.32762489787939796936910852309666e-01 4.32762489787939796936910852309666e-01 8.11643485075055556854373151054460e-03
            4.32762489787939796936910852309666e-01 1.34475020424120406126178295380669e-01 8.11643485075055556854373151054460e-03
            1.34475020424120406126178295380669e-01 4.32762489787939796936910852309666e-01 8.11643485075055556854373151054460e-03
            2.73760447593867972670977906091139e-01 2.73760447593867972670977906091139e-01 9.39944051908108389736806742575936e-03
            2.73760447593867972670977906091139e-01 4.52479104812264054658044187817723e-01 9.39944051908108389736806742575936e-03
            4.52479104812264054658044187817723e-01 2.73760447593867972670977906091139e-01 9.39944051908108389736806742575936e-03
            3.87365851530470428620844813849544e-01 3.87365851530470428620844813849544e-01 9.40865018867229410681751744505164e-03
            3.87365851530470428620844813849544e-01 2.25268296939059142758310372300912e-01 9.40865018867229410681751744505164e-03
            2.25268296939059142758310372300912e-01 3.87365851530470428620844813849544e-01 9.40865018867229410681751744505164e-03
            -1.75486429018026650252654041750588e-04 2.75059108819840636428466495999601e-01 5.02920981817487787110532160284038e-04
            -1.75486429018026650252654041750588e-04 7.25116377609177353846803271153476e-01 5.02920981817487787110532160284038e-04
            2.75059108819840636428466495999601e-01 -1.75486429018026650252654041750588e-04 5.02920981817487787110532160284038e-04
            2.75059108819840636428466495999601e-01 7.25116377609177353846803271153476e-01 5.02920981817487787110532160284038e-04
            7.25116377609177353846803271153476e-01 -1.75486429018026650252654041750588e-04 5.02920981817487787110532160284038e-04
            7.25116377609177353846803271153476e-01 2.75059108819840636428466495999601e-01 5.02920981817487787110532160284038e-04
            7.62723542435699269209647965794829e-03 4.06756516673179857601638786945841e-02 8.88417097597553589299890219876943e-04
            7.62723542435699269209647965794829e-03 9.51697112908325038027612663427135e-01 8.88417097597553589299890219876943e-04
            4.06756516673179857601638786945841e-02 7.62723542435699269209647965794829e-03 8.88417097597553589299890219876943e-04
            4.06756516673179857601638786945841e-02 9.51697112908325038027612663427135e-01 8.88417097597553589299890219876943e-04
            9.51697112908325038027612663427135e-01 7.62723542435699269209647965794829e-03 8.88417097597553589299890219876943e-04
            9.51697112908325038027612663427135e-01 4.06756516673179857601638786945841e-02 8.88417097597553589299890219876943e-04
            7.17445981822133118516182648249924e-03 9.80731978180811037049124934128486e-02 1.26318840233035451989240804948622e-03
            7.17445981822133118516182648249924e-03 8.94752342363697605875927365559619e-01 1.26318840233035451989240804948622e-03
            9.80731978180811037049124934128486e-02 7.17445981822133118516182648249924e-03 1.26318840233035451989240804948622e-03
            9.80731978180811037049124934128486e-02 8.94752342363697605875927365559619e-01 1.26318840233035451989240804948622e-03
            8.94752342363697605875927365559619e-01 7.17445981822133118516182648249924e-03 1.26318840233035451989240804948622e-03
            8.94752342363697605875927365559619e-01 9.80731978180811037049124934128486e-02 1.26318840233035451989240804948622e-03
            6.36745417209535428254563171890368e-03 1.77108966725932348573735453101108e-01 1.44516066721282060436593042140885e-03
            6.36745417209535428254563171890368e-03 8.16523579101972263316611133632250e-01 1.44516066721282060436593042140885e-03
            1.77108966725932348573735453101108e-01 6.36745417209535428254563171890368e-03 1.44516066721282060436593042140885e-03
            1.77108966725932348573735453101108e-01 8.16523579101972263316611133632250e-01 1.44516066721282060436593042140885e-03
            8.16523579101972263316611133632250e-01 6.36745417209535428254563171890368e-03 1.44516066721282060436593042140885e-03
            8.16523579101972263316611133632250e-01 1.77108966725932348573735453101108e-01 1.44516066721282060436593042140885e-03
            7.48751575325660133353178338211364e-03 3.81611685159125635369292695031618e-01 2.15940887549078731899587779707872e-03
            7.48751575325660133353178338211364e-03 6.10900799087617851768072796403430e-01 2.15940887549078731899587779707872e-03
            3.81611685159125635369292695031618e-01 7.48751575325660133353178338211364e-03 2.15940887549078731899587779707872e-03
            3.81611685159125635369292695031618e-01 6.10900799087617851768072796403430e-01 2.15940887549078731899587779707872e-03
            6.10900799087617851768072796403430e-01 7.48751575325660133353178338211364e-03 2.15940887549078731899587779707872e-03
            6.10900799087617851768072796403430e-01 3.81611685159125635369292695031618e-01 2.15940887549078731899587779707872e-03
            3.74639348798537011320242129386315e-02 9.58146591402575242835126800855505e-02 2.81401862144999232243081088711278e-03
            3.74639348798537011320242129386315e-02 8.66721405979888781523357010883046e-01 2.81401862144999232243081088711278e-03
            9.58146591402575242835126800855505e-02 3.74639348798537011320242129386315e-02 2.81401862144999232243081088711278e-03
            9.58146591402575242835126800855505e-02 8.66721405979888781523357010883046e-01 2.81401862144999232243081088711278e-03
            8.66721405979888781523357010883046e-01 3.74639348798537011320242129386315e-02 2.81401862144999232243081088711278e-03
            8.66721405979888781523357010883046e-01 9.58146591402575242835126800855505e-02 2.81401862144999232243081088711278e-03
            1.81035322019235805912629189151630e-02 2.70608545540926448413898697253899e-01 3.11901755421579779045671010351271e-03
            1.81035322019235805912629189151630e-02 7.11287922257149984872626191645395e-01 3.11901755421579779045671010351271e-03
            2.70608545540926448413898697253899e-01 1.81035322019235805912629189151630e-02 3.11901755421579779045671010351271e-03
            2.70608545540926448413898697253899e-01 7.11287922257149984872626191645395e-01 3.11901755421579779045671010351271e-03
            7.11287922257149984872626191645395e-01 1.81035322019235805912629189151630e-02 3.11901755421579779045671010351271e-03
            7.11287922257149984872626191645395e-01 2.70608545540926448413898697253899e-01 3.11901755421579779045671010351271e-03
            3.34959101480053617061471982196963e-02 1.73144453351435828603044342344219e-01 3.26213532770542116723255787746893e-03
            3.34959101480053617061471982196963e-02 7.93359636500558851324171882879455e-01 3.26213532770542116723255787746893e-03
            1.73144453351435828603044342344219e-01 3.34959101480053617061471982196963e-02 3.26213532770542116723255787746893e-03
            1.73144453351435828603044342344219e-01 7.93359636500558851324171882879455e-01 3.26213532770542116723255787746893e-03
            7.93359636500558851324171882879455e-01 3.34959101480053617061471982196963e-02 3.26213532770542116723255787746893e-03
            7.93359636500558851324171882879455e-01 1.73144453351435828603044342344219e-01 3.26213532770542116723255787746893e-03
            3.92067119187980567263096531860356e-02 3.69875713683459339531367504605441e-01 4.78783714906990016946242150197577e-03
            3.92067119187980567263096531860356e-02 5.90917574397742617620110650022980e-01 4.78783714906990016946242150197577e-03
            3.69875713683459339531367504605441e-01 3.92067119187980567263096531860356e-02 4.78783714906990016946242150197577e-03
            3.69875713683459339531367504605441e-01 5.90917574397742617620110650022980e-01 4.78783714906990016946242150197577e-03
            5.90917574397742617620110650022980e-01 3.92067119187980567263096531860356e-02 4.78783714906990016946242150197577e-03
            5.90917574397742617620110650022980e-01 3.69875713683459339531367504605441e-01 4.78783714906990016946242150197577e-03
            8.20162616970389335424584942302317e-02 1.64905597423194411055291652701271e-01 4.83586268905134876450180314577665e-03
            8.20162616970389335424584942302317e-02 7.53078140879766655402249853068497e-01 4.83586268905134876450180314577665e-03
            1.64905597423194411055291652701271e-01 8.20162616970389335424584942302317e-02 4.83586268905134876450180314577665e-03
            1.64905597423194411055291652701271e-01 7.53078140879766655402249853068497e-01 4.83586268905134876450180314577665e-03
            7.53078140879766655402249853068497e-01 8.20162616970389335424584942302317e-02 4.83586268905134876450180314577665e-03
            7.53078140879766655402249853068497e-01 1.64905597423194411055291652701271e-01 4.83586268905134876450180314577665e-03
            6.05057836750334115949456759153691e-02 2.59760411703463867105767803877825e-01 5.26276782211949033241094042523400e-03
            6.05057836750334115949456759153691e-02 6.79733804621502679665923096763436e-01 5.26276782211949033241094042523400e-03
            2.59760411703463867105767803877825e-01 6.05057836750334115949456759153691e-02 5.26276782211949033241094042523400e-03
            2.59760411703463867105767803877825e-01 6.79733804621502679665923096763436e-01 5.26276782211949033241094042523400e-03
            6.79733804621502679665923096763436e-01 6.05057836750334115949456759153691e-02 5.26276782211949033241094042523400e-03
            6.79733804621502679665923096763436e-01 2.59760411703463867105767803877825e-01 5.26276782211949033241094042523400e-03
            1.25482119096221894638532035060052e-01 2.41438155348012661871592854367918e-01 6.88016569707377342174359213800017e-03
            1.25482119096221894638532035060052e-01 6.33079725555765415734299494943116e-01 6.88016569707377342174359213800017e-03
            2.41438155348012661871592854367918e-01 1.25482119096221894638532035060052e-01 6.88016569707377342174359213800017e-03
            2.41438155348012661871592854367918e-01 6.33079725555765415734299494943116e-01 6.88016569707377342174359213800017e-03
            6.33079725555765415734299494943116e-01 1.25482119096221894638532035060052e-01 6.88016569707377342174359213800017e-03
            6.33079725555765415734299494943116e-01 2.41438155348012661871592854367918e-01 6.88016569707377342174359213800017e-03
            9.53671594914007325627380851074122e-02 3.48637419542872584177217731848941e-01 6.93037262232006848888810068842758e-03
            9.53671594914007325627380851074122e-02 5.55995420965726738771195414301474e-01 6.93037262232006848888810068842758e-03
            3.48637419542872584177217731848941e-01 9.53671594914007325627380851074122e-02 6.93037262232006848888810068842758e-03
            3.48637419542872584177217731848941e-01 5.55995420965726738771195414301474e-01 6.93037262232006848888810068842758e-03
            5.55995420965726738771195414301474e-01 9.53671594914007325627380851074122e-02 6.93037262232006848888810068842758e-03
            5.55995420965726738771195414301474e-01 3.48637419542872584177217731848941e-01 6.93037262232006848888810068842758e-03
            1.74659411566860711673143669031560e-01 3.16050027630206786799504925511428e-01 8.52140446786499305975581819438958e-03
            1.74659411566860711673143669031560e-01 5.09290560802932557038502636714838e-01 8.52140446786499305975581819438958e-03
            3.16050027630206786799504925511428e-01 1.74659411566860711673143669031560e-01 8.52140446786499305975581819438958e-03
            3.16050027630206786799504925511428e-01 5.09290560802932557038502636714838e-01 8.52140446786499305975581819438958e-03
            5.09290560802932557038502636714838e-01 1.74659411566860711673143669031560e-01 8.52140446786499305975581819438958e-03
            5.09290560802932557038502636714838e-01 3.16050027630206786799504925511428e-01 8.52140446786499305975581819438958e-03
            ];
        
        
    otherwise
        xyw=[];
end




pointset_stats=[
    15   48 5.41e-16    3    0
    17   58 7.77e-16    0    6
    21   87 5.24e-16    0    0
    22   94 1.63e-15    0    6
    23  102 9.72e-16    0    0
    24  112 1.37e-15    0    0
    25  118 6.75e-16    0    6
    ];




