
function [xyw_bar,pointset_stats]=set_papanicolopulos_A_II_barycentric(deg)

%--------------------------------------------------------------------
% CUBATURE RULE ON THE SIMPLEX [0 0; 0 1; 1 0] WITH ADE EQUAL TO deg.
% THE CUBATURE RULE IS DESCRIBED IN BARYCENTRICAL FORM.
%--------------------------------------------------------------------
% INPUT:
% deg: algebraic degree of precision (ADE) of the rule.
%--------------------------------------------------------------------
% OUTPUT:
% xyw_bar: MATRIX OF NODES xyw(:,1:3) IN BARYCENTRICAL COOR. AND WEIGHTS xyw(:,4).
%      REMARK: The sum of the weights makes 1.
%
% pointset_stats:
%    statistics of the pointset stored in a matrix. Each rows contains
%    degree, cardinality of the full pointset, moment error w.r.t.
%    orthn. Dubiner basis, number of points with negative weights,
%    number of points outside the domain.
%--------------------------------------------------------------------
% SOURCE:
% S-A. Papanicoloupolos
% "New fully symmetric and rotationally symmetric cubature rules on the
% triangle using minimal orthonormal bases".
% JCAM 294 (2016), 39-48.
% ** Using file fullsymmetry.txt from archives.
%--------------------------------------------------------------------
% | DEG | CARD | W- | PT.O|   M.E.   |Q.|
%--------------------------------------------------------------------
% |  22 |  96 |   0 |   0 | 6.73e-16 |PI|
% |  25 | 120 |   6 |   0 | 7.55e-15 |NI|
%--------------------------------------------------------------------
% LEGEND:
%--------------------------------------------------------------------
% DEG  : algebraic degree of precision
% CARD : cardinality of the pointset
% W-   : negative weights
% PT.O : points outside the simplex [0 0; 1 0; 0 1].
% M.E. : maximum moment error in inf norm.
% Q.: quality of the rule.
%        PI: pos. weights, points in simplex
%        NI: some neg. weights, points in simplex
%        P0: pos. weights, some points out of the simplex
%        N0: some neg. weights, some points out of the simplex
%--------------------------------------------------------------------



switch deg
    
    case  22
        
        % ALG. DEG.:   22
        % PTS CARD.:   96
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.7e-16
        
        xyw_bar=[
            7.22823551590805277872187062371268e-03 7.22823551590805277872187062371268e-03 9.85543528968183868421704119100468e-01 6.97145696250740581553784203094892e-04
            7.22823551590805277872187062371268e-03 9.85543528968183868421704119100468e-01 7.22823551590805277872187062371268e-03 6.97145696250740581553784203094892e-04
            9.85543528968183868421704119100468e-01 7.22823551590805277872187062371268e-03 7.22823551590805277872187062371268e-03 6.97145696250740581553784203094892e-04
            4.99714916562231881158595570013858e-01 4.99714916562231881158595570013858e-01 5.70166875536237682808859972283244e-04 1.73365376198036350192943633174991e-03
            4.99714916562231881158595570013858e-01 5.70166875536237682808859972283244e-04 4.99714916562231881158595570013858e-01 1.73365376198036350192943633174991e-03
            5.70166875536237682808859972283244e-04 4.99714916562231881158595570013858e-01 4.99714916562231881158595570013858e-01 1.73365376198036350192943633174991e-03
            4.19525272728040882208588868707011e-02 4.19525272728040882208588868707011e-02 9.16094945454391851313857841887511e-01 4.66131304304557517875773342552748e-03
            4.19525272728040882208588868707011e-02 9.16094945454391851313857841887511e-01 4.19525272728040882208588868707011e-02 4.66131304304557517875773342552748e-03
            9.16094945454391851313857841887511e-01 4.19525272728040882208588868707011e-02 4.19525272728040882208588868707011e-02 4.66131304304557517875773342552748e-03
            3.14132191520500925197012520584394e-01 3.14132191520500925197012520584394e-01 3.71735616958998149605974958831212e-01 1.15845767655995707939853289758503e-02
            3.14132191520500925197012520584394e-01 3.71735616958998149605974958831212e-01 3.14132191520500925197012520584394e-01 1.15845767655995707939853289758503e-02
            3.71735616958998149605974958831212e-01 3.14132191520500925197012520584394e-01 3.14132191520500925197012520584394e-01 1.15845767655995707939853289758503e-02
            4.80013651617116565706311348549207e-01 4.80013651617116565706311348549207e-01 3.99726967657668685873773029015865e-02 1.18789740330219155406421549514562e-02
            4.80013651617116565706311348549207e-01 3.99726967657668685873773029015865e-02 4.80013651617116565706311348549207e-01 1.18789740330219155406421549514562e-02
            3.99726967657668685873773029015865e-02 4.80013651617116565706311348549207e-01 4.80013651617116565706311348549207e-01 1.18789740330219155406421549514562e-02
            1.23087454230161152946010361119988e-01 1.23087454230161152946010361119988e-01 7.53825091539677694107979277760023e-01 1.53388036556667065046788422932877e-02
            1.23087454230161152946010361119988e-01 7.53825091539677694107979277760023e-01 1.23087454230161152946010361119988e-01 1.53388036556667065046788422932877e-02
            7.53825091539677694107979277760023e-01 1.23087454230161152946010361119988e-01 1.23087454230161152946010361119988e-01 1.53388036556667065046788422932877e-02
            4.45719630439115332976030003919732e-01 4.45719630439115332976030003919732e-01 1.08560739121769334047939992160536e-01 1.97163769402723468293636699399940e-02
            4.45719630439115332976030003919732e-01 1.08560739121769334047939992160536e-01 4.45719630439115332976030003919732e-01 1.97163769402723468293636699399940e-02
            1.08560739121769334047939992160536e-01 4.45719630439115332976030003919732e-01 4.45719630439115332976030003919732e-01 1.97163769402723468293636699399940e-02
            1.89862412183893686234981146299106e-01 1.89862412183893686234981146299106e-01 6.20275175632212683041188938659616e-01 2.05721514917347643447431693175531e-02
            1.89862412183893686234981146299106e-01 6.20275175632212683041188938659616e-01 1.89862412183893686234981146299106e-01 2.05721514917347643447431693175531e-02
            6.20275175632212683041188938659616e-01 1.89862412183893686234981146299106e-01 1.89862412183893686234981146299106e-01 2.05721514917347643447431693175531e-02
            2.60339233381755796870038466295227e-01 2.60339233381755796870038466295227e-01 4.79321533236488406259923067409545e-01 2.38406602781370477450728628809884e-02
            2.60339233381755796870038466295227e-01 4.79321533236488406259923067409545e-01 2.60339233381755796870038466295227e-01 2.38406602781370477450728628809884e-02
            4.79321533236488406259923067409545e-01 2.60339233381755796870038466295227e-01 2.60339233381755796870038466295227e-01 2.38406602781370477450728628809884e-02
            3.96366847671625999183220301347319e-01 3.96366847671625999183220301347319e-01 2.07266304656748001633559397305362e-01 2.44611970898748823377566452563769e-02
            3.96366847671625999183220301347319e-01 2.07266304656748001633559397305362e-01 3.96366847671625999183220301347319e-01 2.44611970898748823377566452563769e-02
            2.07266304656748001633559397305362e-01 3.96366847671625999183220301347319e-01 3.96366847671625999183220301347319e-01 2.44611970898748823377566452563769e-02
            7.73958468860307807757337172915868e-03 3.91001678273283775899571423906309e-02 9.53160247484068579026939005416352e-01 1.82862802731121004522207851294979e-03
            7.73958468860307807757337172915868e-03 9.53160247484068579026939005416352e-01 3.91001678273283775899571423906309e-02 1.82862802731121004522207851294979e-03
            3.91001678273283775899571423906309e-02 7.73958468860307807757337172915868e-03 9.53160247484068579026939005416352e-01 1.82862802731121004522207851294979e-03
            3.91001678273283775899571423906309e-02 9.53160247484068579026939005416352e-01 7.73958468860307807757337172915868e-03 1.82862802731121004522207851294979e-03
            9.53160247484068579026939005416352e-01 7.73958468860307807757337172915868e-03 3.91001678273283775899571423906309e-02 1.82862802731121004522207851294979e-03
            9.53160247484068579026939005416352e-01 3.91001678273283775899571423906309e-02 7.73958468860307807757337172915868e-03 1.82862802731121004522207851294979e-03
            5.77064130218403622228640070801475e-04 2.99563010316006073097128137305845e-01 6.99859925553775519269095184426988e-01 1.88044970396283915928248298143899e-03
            5.77064130218403622228640070801475e-04 6.99859925553775519269095184426988e-01 2.99563010316006073097128137305845e-01 1.88044970396283915928248298143899e-03
            2.99563010316006073097128137305845e-01 5.77064130218403622228640070801475e-04 6.99859925553775519269095184426988e-01 1.88044970396283915928248298143899e-03
            2.99563010316006073097128137305845e-01 6.99859925553775519269095184426988e-01 5.77064130218403622228640070801475e-04 1.88044970396283915928248298143899e-03
            6.99859925553775519269095184426988e-01 5.77064130218403622228640070801475e-04 2.99563010316006073097128137305845e-01 1.88044970396283915928248298143899e-03
            6.99859925553775519269095184426988e-01 2.99563010316006073097128137305845e-01 5.77064130218403622228640070801475e-04 1.88044970396283915928248298143899e-03
            9.79577185352314104560544905098141e-03 9.91675808522524682153687081154203e-02 8.91036647294224382065408462949563e-01 3.70410864001110336912736542558378e-03
            9.79577185352314104560544905098141e-03 8.91036647294224382065408462949563e-01 9.91675808522524682153687081154203e-02 3.70410864001110336912736542558378e-03
            9.91675808522524682153687081154203e-02 9.79577185352314104560544905098141e-03 8.91036647294224382065408462949563e-01 3.70410864001110336912736542558378e-03
            9.91675808522524682153687081154203e-02 8.91036647294224382065408462949563e-01 9.79577185352314104560544905098141e-03 3.70410864001110336912736542558378e-03
            8.91036647294224382065408462949563e-01 9.79577185352314104560544905098141e-03 9.91675808522524682153687081154203e-02 3.70410864001110336912736542558378e-03
            8.91036647294224382065408462949563e-01 9.91675808522524682153687081154203e-02 9.79577185352314104560544905098141e-03 3.70410864001110336912736542558378e-03
            9.18868020649896491414132526642788e-03 1.86190681552805714638054723764071e-01 8.04620638240695296161675287294202e-01 4.66834037183480772548183068693106e-03
            9.18868020649896491414132526642788e-03 8.04620638240695296161675287294202e-01 1.86190681552805714638054723764071e-01 4.66834037183480772548183068693106e-03
            1.86190681552805714638054723764071e-01 9.18868020649896491414132526642788e-03 8.04620638240695296161675287294202e-01 4.66834037183480772548183068693106e-03
            1.86190681552805714638054723764071e-01 8.04620638240695296161675287294202e-01 9.18868020649896491414132526642788e-03 4.66834037183480772548183068693106e-03
            8.04620638240695296161675287294202e-01 9.18868020649896491414132526642788e-03 1.86190681552805714638054723764071e-01 4.66834037183480772548183068693106e-03
            8.04620638240695296161675287294202e-01 1.86190681552805714638054723764071e-01 9.18868020649896491414132526642788e-03 4.66834037183480772548183068693106e-03
            1.34670875684018177820266970456942e-02 4.02938891855100456140803544258233e-01 5.83594020576497674035465479391860e-01 7.21134117192958907366362453217334e-03
            1.34670875684018177820266970456942e-02 5.83594020576497674035465479391860e-01 4.02938891855100456140803544258233e-01 7.21134117192958907366362453217334e-03
            4.02938891855100456140803544258233e-01 1.34670875684018177820266970456942e-02 5.83594020576497674035465479391860e-01 7.21134117192958907366362453217334e-03
            4.02938891855100456140803544258233e-01 5.83594020576497674035465479391860e-01 1.34670875684018177820266970456942e-02 7.21134117192958907366362453217334e-03
            5.83594020576497674035465479391860e-01 1.34670875684018177820266970456942e-02 4.02938891855100456140803544258233e-01 7.21134117192958907366362453217334e-03
            5.83594020576497674035465479391860e-01 4.02938891855100456140803544258233e-01 1.34670875684018177820266970456942e-02 7.21134117192958907366362453217334e-03
            5.06263124673368980488419310859172e-02 1.03588085409291874583459502900951e-01 8.45785602123371171856547334755305e-01 7.96743335988734681374356938476922e-03
            5.06263124673368980488419310859172e-02 8.45785602123371171856547334755305e-01 1.03588085409291874583459502900951e-01 7.96743335988734681374356938476922e-03
            1.03588085409291874583459502900951e-01 5.06263124673368980488419310859172e-02 8.45785602123371171856547334755305e-01 7.96743335988734681374356938476922e-03
            1.03588085409291874583459502900951e-01 8.45785602123371171856547334755305e-01 5.06263124673368980488419310859172e-02 7.96743335988734681374356938476922e-03
            8.45785602123371171856547334755305e-01 5.06263124673368980488419310859172e-02 1.03588085409291874583459502900951e-01 7.96743335988734681374356938476922e-03
            8.45785602123371171856547334755305e-01 1.03588085409291874583459502900951e-01 5.06263124673368980488419310859172e-02 7.96743335988734681374356938476922e-03
            4.83881177618376498439367594528449e-02 1.80823006888328891017536648178066e-01 7.70788875349833535466359535348602e-01 9.03580036943713894859175894680448e-03
            4.83881177618376498439367594528449e-02 7.70788875349833535466359535348602e-01 1.80823006888328891017536648178066e-01 9.03580036943713894859175894680448e-03
            1.80823006888328891017536648178066e-01 4.83881177618376498439367594528449e-02 7.70788875349833535466359535348602e-01 9.03580036943713894859175894680448e-03
            1.80823006888328891017536648178066e-01 7.70788875349833535466359535348602e-01 4.83881177618376498439367594528449e-02 9.03580036943713894859175894680448e-03
            7.70788875349833535466359535348602e-01 4.83881177618376498439367594528449e-02 1.80823006888328891017536648178066e-01 9.03580036943713894859175894680448e-03
            7.70788875349833535466359535348602e-01 1.80823006888328891017536648178066e-01 4.83881177618376498439367594528449e-02 9.03580036943713894859175894680448e-03
            2.74914358532603458895593462329998e-02 2.81943415079491888075580163786071e-01 6.90565149067247752157072682166472e-01 9.48064111155901868832618362148423e-03
            2.74914358532603458895593462329998e-02 6.90565149067247752157072682166472e-01 2.81943415079491888075580163786071e-01 9.48064111155901868832618362148423e-03
            2.81943415079491888075580163786071e-01 2.74914358532603458895593462329998e-02 6.90565149067247752157072682166472e-01 9.48064111155901868832618362148423e-03
            2.81943415079491888075580163786071e-01 6.90565149067247752157072682166472e-01 2.74914358532603458895593462329998e-02 9.48064111155901868832618362148423e-03
            6.90565149067247752157072682166472e-01 2.74914358532603458895593462329998e-02 2.81943415079491888075580163786071e-01 9.48064111155901868832618362148423e-03
            6.90565149067247752157072682166472e-01 2.81943415079491888075580163786071e-01 2.74914358532603458895593462329998e-02 9.48064111155901868832618362148423e-03
            6.71339744477043864900878133994411e-02 3.50712853397991353165963346327771e-01 5.82153172154304288099524455901701e-01 1.56173701283532154193700591804372e-02
            6.71339744477043864900878133994411e-02 5.82153172154304288099524455901701e-01 3.50712853397991353165963346327771e-01 1.56173701283532154193700591804372e-02
            3.50712853397991353165963346327771e-01 6.71339744477043864900878133994411e-02 5.82153172154304288099524455901701e-01 1.56173701283532154193700591804372e-02
            3.50712853397991353165963346327771e-01 5.82153172154304288099524455901701e-01 6.71339744477043864900878133994411e-02 1.56173701283532154193700591804372e-02
            5.82153172154304288099524455901701e-01 6.71339744477043864900878133994411e-02 3.50712853397991353165963346327771e-01 1.56173701283532154193700591804372e-02
            5.82153172154304288099524455901701e-01 3.50712853397991353165963346327771e-01 6.71339744477043864900878133994411e-02 1.56173701283532154193700591804372e-02
            9.67043036168603459268311439700483e-02 2.30176404403601320147743081179215e-01 6.73119291979538347803213582665194e-01 1.62453191810569899655725123466254e-02
            9.67043036168603459268311439700483e-02 6.73119291979538347803213582665194e-01 2.30176404403601320147743081179215e-01 1.62453191810569899655725123466254e-02
            2.30176404403601320147743081179215e-01 9.67043036168603459268311439700483e-02 6.73119291979538347803213582665194e-01 1.62453191810569899655725123466254e-02
            2.30176404403601320147743081179215e-01 6.73119291979538347803213582665194e-01 9.67043036168603459268311439700483e-02 1.62453191810569899655725123466254e-02
            6.73119291979538347803213582665194e-01 9.67043036168603459268311439700483e-02 2.30176404403601320147743081179215e-01 1.62453191810569899655725123466254e-02
            6.73119291979538347803213582665194e-01 2.30176404403601320147743081179215e-01 9.67043036168603459268311439700483e-02 1.62453191810569899655725123466254e-02
            1.51930454826702282522532527764270e-01 3.11593235863486472947414540612954e-01 5.36476309309811272285628547251690e-01 2.17848082235315224086669161351892e-02
            1.51930454826702282522532527764270e-01 5.36476309309811272285628547251690e-01 3.11593235863486472947414540612954e-01 2.17848082235315224086669161351892e-02
            3.11593235863486472947414540612954e-01 1.51930454826702282522532527764270e-01 5.36476309309811272285628547251690e-01 2.17848082235315224086669161351892e-02
            3.11593235863486472947414540612954e-01 5.36476309309811272285628547251690e-01 1.51930454826702282522532527764270e-01 2.17848082235315224086669161351892e-02
            5.36476309309811272285628547251690e-01 1.51930454826702282522532527764270e-01 3.11593235863486472947414540612954e-01 2.17848082235315224086669161351892e-02
            5.36476309309811272285628547251690e-01 3.11593235863486472947414540612954e-01 1.51930454826702282522532527764270e-01 2.17848082235315224086669161351892e-02
            ];
        
    case  25
        
        % ALG. DEG.:   25
        % PTS CARD.:  120
        % NEG. W.  :    6
        % OUT PTS. :    0
        % M.E.INF. : 7.5e-15
        
        xyw_bar=[
            2.41705564997226424184262327798933e-01 2.41705564997226424184262327798933e-01 5.16588870005547207142626575659961e-01 -2.69549612236607893933637569716666e+00
            2.41705564997226424184262327798933e-01 5.16588870005547207142626575659961e-01 2.41705564997226424184262327798933e-01 -2.69549612236607893933637569716666e+00
            5.16588870005547207142626575659961e-01 2.41705564997226424184262327798933e-01 2.41705564997226424184262327798933e-01 -2.69549612236607893933637569716666e+00
            6.68411454103012420135954130273603e-02 6.68411454103012420135954130273603e-02 8.66317709179397543728384789574193e-01 -1.80530872827312717683601039908581e-03
            6.68411454103012420135954130273603e-02 8.66317709179397543728384789574193e-01 6.68411454103012420135954130273603e-02 -1.80530872827312717683601039908581e-03
            8.66317709179397543728384789574193e-01 6.68411454103012420135954130273603e-02 6.68411454103012420135954130273603e-02 -1.80530872827312717683601039908581e-03
            7.91213740209865609709360967372049e-03 7.91213740209865609709360967372049e-03 9.84175725195802653111343261116417e-01 8.18660001776079851443235213537264e-04
            7.91213740209865609709360967372049e-03 9.84175725195802653111343261116417e-01 7.91213740209865609709360967372049e-03 8.18660001776079851443235213537264e-04
            9.84175725195802653111343261116417e-01 7.91213740209865609709360967372049e-03 7.91213740209865609709360967372049e-03 8.18660001776079851443235213537264e-04
            3.86825002238830387146784062224469e-02 3.86825002238830387146784062224469e-02 9.22634999552233936448430995369563e-01 3.87240802246695140290122516546489e-03
            3.86825002238830387146784062224469e-02 9.22634999552233936448430995369563e-01 3.86825002238830387146784062224469e-02 3.87240802246695140290122516546489e-03
            9.22634999552233936448430995369563e-01 3.86825002238830387146784062224469e-02 3.86825002238830387146784062224469e-02 3.87240802246695140290122516546489e-03
            4.95717425830097813133079398539849e-01 4.95717425830097813133079398539849e-01 8.56514833980437373384120292030275e-03 5.19562728909421598505469219730912e-03
            4.95717425830097813133079398539849e-01 8.56514833980437373384120292030275e-03 4.95717425830097813133079398539849e-01 5.19562728909421598505469219730912e-03
            8.56514833980437373384120292030275e-03 4.95717425830097813133079398539849e-01 4.95717425830097813133079398539849e-01 5.19562728909421598505469219730912e-03
            9.66934516840111946134328491098131e-02 9.66934516840111946134328491098131e-02 8.06613096631977555261983070522547e-01 8.27582560362955486910063740424448e-03
            9.66934516840111946134328491098131e-02 8.06613096631977555261983070522547e-01 9.66934516840111946134328491098131e-02 8.27582560362955486910063740424448e-03
            8.06613096631977555261983070522547e-01 9.66934516840111946134328491098131e-02 9.66934516840111946134328491098131e-02 8.27582560362955486910063740424448e-03
            4.76796052923181057359158785402542e-01 4.76796052923181057359158785402542e-01 4.64078941536378852816824291949160e-02 1.14946929251025162233013077184296e-02
            4.76796052923181057359158785402542e-01 4.64078941536378852816824291949160e-02 4.76796052923181057359158785402542e-01 1.14946929251025162233013077184296e-02
            4.64078941536378852816824291949160e-02 4.76796052923181057359158785402542e-01 4.76796052923181057359158785402542e-01 1.14946929251025162233013077184296e-02
            1.51363116253387219867221347158193e-01 1.51363116253387219867221347158193e-01 6.97273767493225560265557305683615e-01 1.42897587003528812182873153346918e-02
            1.51363116253387219867221347158193e-01 6.97273767493225560265557305683615e-01 1.51363116253387219867221347158193e-01 1.42897587003528812182873153346918e-02
            6.97273767493225560265557305683615e-01 1.51363116253387219867221347158193e-01 1.51363116253387219867221347158193e-01 1.42897587003528812182873153346918e-02
            4.44551436843102165674679326912155e-01 4.44551436843102165674679326912155e-01 1.10897126313795668650641346175689e-01 1.55950327338980846564320259517444e-02
            4.44551436843102165674679326912155e-01 1.10897126313795668650641346175689e-01 4.44551436843102165674679326912155e-01 1.55950327338980846564320259517444e-02
            1.10897126313795668650641346175689e-01 4.44551436843102165674679326912155e-01 4.44551436843102165674679326912155e-01 1.55950327338980846564320259517444e-02
            3.61911501476409047661775275628315e-01 3.61911501476409047661775275628315e-01 2.76176997047181904676449448743369e-01 1.70629598758336893660203514855311e-02
            3.61911501476409047661775275628315e-01 2.76176997047181904676449448743369e-01 3.61911501476409047661775275628315e-01 1.70629598758336893660203514855311e-02
            2.76176997047181904676449448743369e-01 3.61911501476409047661775275628315e-01 3.61911501476409047661775275628315e-01 1.70629598758336893660203514855311e-02
            4.01885652208385124595224624499679e-01 4.01885652208385124595224624499679e-01 1.96228695583229750809550751000643e-01 1.71270723975187437460032668923304e-02
            4.01885652208385124595224624499679e-01 1.96228695583229750809550751000643e-01 4.01885652208385124595224624499679e-01 1.71270723975187437460032668923304e-02
            1.96228695583229750809550751000643e-01 4.01885652208385124595224624499679e-01 4.01885652208385124595224624499679e-01 1.71270723975187437460032668923304e-02
            2.29157170026697831755058132330305e-01 2.29157170026697831755058132330305e-01 5.41685659946604336489883735339390e-01 1.43068617453155716434665123415471e-01
            2.29157170026697831755058132330305e-01 5.41685659946604336489883735339390e-01 2.29157170026697831755058132330305e-01 1.43068617453155716434665123415471e-01
            5.41685659946604336489883735339390e-01 2.29157170026697831755058132330305e-01 2.29157170026697831755058132330305e-01 1.43068617453155716434665123415471e-01
            3.03302471090682499409163508374832e-04 1.78730929732423882150271765567595e-01 8.20965767796485446083920578530524e-01 1.13714181496523607066417849154050e-03
            3.03302471090682499409163508374832e-04 8.20965767796485446083920578530524e-01 1.78730929732423882150271765567595e-01 1.13714181496523607066417849154050e-03
            1.78730929732423882150271765567595e-01 3.03302471090682499409163508374832e-04 8.20965767796485446083920578530524e-01 1.13714181496523607066417849154050e-03
            1.78730929732423882150271765567595e-01 8.20965767796485446083920578530524e-01 3.03302471090682499409163508374832e-04 1.13714181496523607066417849154050e-03
            8.20965767796485446083920578530524e-01 3.03302471090682499409163508374832e-04 1.78730929732423882150271765567595e-01 1.13714181496523607066417849154050e-03
            8.20965767796485446083920578530524e-01 1.78730929732423882150271765567595e-01 3.03302471090682499409163508374832e-04 1.13714181496523607066417849154050e-03
            7.28506607201764421616330125175409e-03 4.10919626179091224127404302635114e-02 9.51622971310073229034287578542717e-01 1.71355194199544259327228079570204e-03
            7.28506607201764421616330125175409e-03 9.51622971310073229034287578542717e-01 4.10919626179091224127404302635114e-02 1.71355194199544259327228079570204e-03
            4.10919626179091224127404302635114e-02 7.28506607201764421616330125175409e-03 9.51622971310073229034287578542717e-01 1.71355194199544259327228079570204e-03
            4.10919626179091224127404302635114e-02 9.51622971310073229034287578542717e-01 7.28506607201764421616330125175409e-03 1.71355194199544259327228079570204e-03
            9.51622971310073229034287578542717e-01 7.28506607201764421616330125175409e-03 4.10919626179091224127404302635114e-02 1.71355194199544259327228079570204e-03
            9.51622971310073229034287578542717e-01 4.10919626179091224127404302635114e-02 7.28506607201764421616330125175409e-03 1.71355194199544259327228079570204e-03
            2.09819055278945201109541685013937e-03 3.80405002447553652888956321476144e-01 6.17496806999656921988162139314227e-01 1.96815375779925243182599814417699e-03
            2.09819055278945201109541685013937e-03 6.17496806999656921988162139314227e-01 3.80405002447553652888956321476144e-01 1.96815375779925243182599814417699e-03
            3.80405002447553652888956321476144e-01 2.09819055278945201109541685013937e-03 6.17496806999656921988162139314227e-01 1.96815375779925243182599814417699e-03
            3.80405002447553652888956321476144e-01 6.17496806999656921988162139314227e-01 2.09819055278945201109541685013937e-03 1.96815375779925243182599814417699e-03
            6.17496806999656921988162139314227e-01 2.09819055278945201109541685013937e-03 3.80405002447553652888956321476144e-01 1.96815375779925243182599814417699e-03
            6.17496806999656921988162139314227e-01 3.80405002447553652888956321476144e-01 2.09819055278945201109541685013937e-03 1.96815375779925243182599814417699e-03
            8.24935540216329907248216102289007e-03 9.84515981635840919938829074453679e-02 8.93299046434252619341975787392585e-01 2.89806198086503671515479219067402e-03
            8.24935540216329907248216102289007e-03 8.93299046434252619341975787392585e-01 9.84515981635840919938829074453679e-02 2.89806198086503671515479219067402e-03
            9.84515981635840919938829074453679e-02 8.24935540216329907248216102289007e-03 8.93299046434252619341975787392585e-01 2.89806198086503671515479219067402e-03
            9.84515981635840919938829074453679e-02 8.93299046434252619341975787392585e-01 8.24935540216329907248216102289007e-03 2.89806198086503671515479219067402e-03
            8.93299046434252619341975787392585e-01 8.24935540216329907248216102289007e-03 9.84515981635840919938829074453679e-02 2.89806198086503671515479219067402e-03
            8.93299046434252619341975787392585e-01 9.84515981635840919938829074453679e-02 8.24935540216329907248216102289007e-03 2.89806198086503671515479219067402e-03
            9.19844896889114756188821075966189e-03 2.71278993067566198682527556229616e-01 7.19522557963542697123671132430900e-01 4.77607271269491970300258287807083e-03
            9.19844896889114756188821075966189e-03 7.19522557963542697123671132430900e-01 2.71278993067566198682527556229616e-01 4.77607271269491970300258287807083e-03
            2.71278993067566198682527556229616e-01 9.19844896889114756188821075966189e-03 7.19522557963542697123671132430900e-01 4.77607271269491970300258287807083e-03
            2.71278993067566198682527556229616e-01 7.19522557963542697123671132430900e-01 9.19844896889114756188821075966189e-03 4.77607271269491970300258287807083e-03
            7.19522557963542697123671132430900e-01 9.19844896889114756188821075966189e-03 2.71278993067566198682527556229616e-01 4.77607271269491970300258287807083e-03
            7.19522557963542697123671132430900e-01 2.71278993067566198682527556229616e-01 9.19844896889114756188821075966189e-03 4.77607271269491970300258287807083e-03
            2.27894351251585400297550165760185e-02 1.73042709639871727933524425679934e-01 8.04167855234969808364553500723559e-01 6.48989797457108063410968767925624e-03
            2.27894351251585400297550165760185e-02 8.04167855234969808364553500723559e-01 1.73042709639871727933524425679934e-01 6.48989797457108063410968767925624e-03
            1.73042709639871727933524425679934e-01 2.27894351251585400297550165760185e-02 8.04167855234969808364553500723559e-01 6.48989797457108063410968767925624e-03
            1.73042709639871727933524425679934e-01 8.04167855234969808364553500723559e-01 2.27894351251585400297550165760185e-02 6.48989797457108063410968767925624e-03
            8.04167855234969808364553500723559e-01 2.27894351251585400297550165760185e-02 1.73042709639871727933524425679934e-01 6.48989797457108063410968767925624e-03
            8.04167855234969808364553500723559e-01 1.73042709639871727933524425679934e-01 2.27894351251585400297550165760185e-02 6.48989797457108063410968767925624e-03
            4.38684606504693083706136746968696e-02 9.17541328991964832972882959438721e-02 8.64377406450334229148779741080943e-01 6.84104347762788406878131652888442e-03
            4.38684606504693083706136746968696e-02 8.64377406450334229148779741080943e-01 9.17541328991964832972882959438721e-02 6.84104347762788406878131652888442e-03
            9.17541328991964832972882959438721e-02 4.38684606504693083706136746968696e-02 8.64377406450334229148779741080943e-01 6.84104347762788406878131652888442e-03
            9.17541328991964832972882959438721e-02 8.64377406450334229148779741080943e-01 4.38684606504693083706136746968696e-02 6.84104347762788406878131652888442e-03
            8.64377406450334229148779741080943e-01 4.38684606504693083706136746968696e-02 9.17541328991964832972882959438721e-02 6.84104347762788406878131652888442e-03
            8.64377406450334229148779741080943e-01 9.17541328991964832972882959438721e-02 4.38684606504693083706136746968696e-02 6.84104347762788406878131652888442e-03
            2.48398299166222302769568841540604e-02 3.76811651708434280916293346308521e-01 5.98348518374943516562325385166332e-01 8.26473228086356190291716927731613e-03
            2.48398299166222302769568841540604e-02 5.98348518374943516562325385166332e-01 3.76811651708434280916293346308521e-01 8.26473228086356190291716927731613e-03
            3.76811651708434280916293346308521e-01 2.48398299166222302769568841540604e-02 5.98348518374943516562325385166332e-01 8.26473228086356190291716927731613e-03
            3.76811651708434280916293346308521e-01 5.98348518374943516562325385166332e-01 2.48398299166222302769568841540604e-02 8.26473228086356190291716927731613e-03
            5.98348518374943516562325385166332e-01 2.48398299166222302769568841540604e-02 3.76811651708434280916293346308521e-01 8.26473228086356190291716927731613e-03
            5.98348518374943516562325385166332e-01 3.76811651708434280916293346308521e-01 2.48398299166222302769568841540604e-02 8.26473228086356190291716927731613e-03
            4.63554300688644033145990874800191e-02 2.65561584563092811794859926521895e-01 6.88082985368042798768328793812543e-01 1.02243648962691592585461108910749e-02
            4.63554300688644033145990874800191e-02 6.88082985368042798768328793812543e-01 2.65561584563092811794859926521895e-01 1.02243648962691592585461108910749e-02
            2.65561584563092811794859926521895e-01 4.63554300688644033145990874800191e-02 6.88082985368042798768328793812543e-01 1.02243648962691592585461108910749e-02
            2.65561584563092811794859926521895e-01 6.88082985368042798768328793812543e-01 4.63554300688644033145990874800191e-02 1.02243648962691592585461108910749e-02
            6.88082985368042798768328793812543e-01 4.63554300688644033145990874800191e-02 2.65561584563092811794859926521895e-01 1.02243648962691592585461108910749e-02
            6.88082985368042798768328793812543e-01 2.65561584563092811794859926521895e-01 4.63554300688644033145990874800191e-02 1.02243648962691592585461108910749e-02
            7.19441719771225091095701031917997e-02 1.70539121421737188688538822134433e-01 7.57516706601140232812952035601484e-01 1.02441540629609941642774728620680e-02
            7.19441719771225091095701031917997e-02 7.57516706601140232812952035601484e-01 1.70539121421737188688538822134433e-01 1.02441540629609941642774728620680e-02
            1.70539121421737188688538822134433e-01 7.19441719771225091095701031917997e-02 7.57516706601140232812952035601484e-01 1.02441540629609941642774728620680e-02
            1.70539121421737188688538822134433e-01 7.57516706601140232812952035601484e-01 7.19441719771225091095701031917997e-02 1.02441540629609941642774728620680e-02
            7.57516706601140232812952035601484e-01 7.19441719771225091095701031917997e-02 1.70539121421737188688538822134433e-01 1.02441540629609941642774728620680e-02
            7.57516706601140232812952035601484e-01 1.70539121421737188688538822134433e-01 7.19441719771225091095701031917997e-02 1.02441540629609941642774728620680e-02
            7.71705874722226348172426924065803e-02 3.55127335899842411226501326382277e-01 5.67702076627935037222982828097884e-01 1.36719002036591701088052985824106e-02
            7.71705874722226348172426924065803e-02 5.67702076627935037222982828097884e-01 3.55127335899842411226501326382277e-01 1.36719002036591701088052985824106e-02
            3.55127335899842411226501326382277e-01 7.71705874722226348172426924065803e-02 5.67702076627935037222982828097884e-01 1.36719002036591701088052985824106e-02
            3.55127335899842411226501326382277e-01 5.67702076627935037222982828097884e-01 7.71705874722226348172426924065803e-02 1.36719002036591701088052985824106e-02
            5.67702076627935037222982828097884e-01 7.71705874722226348172426924065803e-02 3.55127335899842411226501326382277e-01 1.36719002036591701088052985824106e-02
            5.67702076627935037222982828097884e-01 3.55127335899842411226501326382277e-01 7.71705874722226348172426924065803e-02 1.36719002036591701088052985824106e-02
            1.14037668606329273268684687536734e-01 2.44672816769111428625294024641335e-01 6.41289514624559298106021287821932e-01 1.50578629356946779721537765794892e-02
            1.14037668606329273268684687536734e-01 6.41289514624559298106021287821932e-01 2.44672816769111428625294024641335e-01 1.50578629356946779721537765794892e-02
            2.44672816769111428625294024641335e-01 1.14037668606329273268684687536734e-01 6.41289514624559298106021287821932e-01 1.50578629356946779721537765794892e-02
            2.44672816769111428625294024641335e-01 6.41289514624559298106021287821932e-01 1.14037668606329273268684687536734e-01 1.50578629356946779721537765794892e-02
            6.41289514624559298106021287821932e-01 1.14037668606329273268684687536734e-01 2.44672816769111428625294024641335e-01 1.50578629356946779721537765794892e-02
            6.41289514624559298106021287821932e-01 2.44672816769111428625294024641335e-01 1.14037668606329273268684687536734e-01 1.50578629356946779721537765794892e-02
            1.53156702364264596383236494148150e-01 3.28189451188987735275048862604308e-01 5.18653846446747612830563411989715e-01 1.58064259318788091768226422573207e-02
            1.53156702364264596383236494148150e-01 5.18653846446747612830563411989715e-01 3.28189451188987735275048862604308e-01 1.58064259318788091768226422573207e-02
            3.28189451188987735275048862604308e-01 1.53156702364264596383236494148150e-01 5.18653846446747612830563411989715e-01 1.58064259318788091768226422573207e-02
            3.28189451188987735275048862604308e-01 5.18653846446747612830563411989715e-01 1.53156702364264596383236494148150e-01 1.58064259318788091768226422573207e-02
            5.18653846446747612830563411989715e-01 1.53156702364264596383236494148150e-01 3.28189451188987735275048862604308e-01 1.58064259318788091768226422573207e-02
            5.18653846446747612830563411989715e-01 3.28189451188987735275048862604308e-01 1.53156702364264596383236494148150e-01 1.58064259318788091768226422573207e-02
            2.39691112819588936222459096825332e-01 2.45045069544505583758464695165458e-01 5.15263817635905563285803054895950e-01 1.29782369074058179236885735008400e+00
            2.39691112819588936222459096825332e-01 5.15263817635905563285803054895950e-01 2.45045069544505583758464695165458e-01 1.29782369074058179236885735008400e+00
            2.45045069544505583758464695165458e-01 2.39691112819588936222459096825332e-01 5.15263817635905563285803054895950e-01 1.29782369074058179236885735008400e+00
            2.45045069544505583758464695165458e-01 5.15263817635905563285803054895950e-01 2.39691112819588936222459096825332e-01 1.29782369074058179236885735008400e+00
            5.15263817635905563285803054895950e-01 2.39691112819588936222459096825332e-01 2.45045069544505583758464695165458e-01 1.29782369074058179236885735008400e+00
            5.15263817635905563285803054895950e-01 2.45045069544505583758464695165458e-01 2.39691112819588936222459096825332e-01 1.29782369074058179236885735008400e+00
            ];
        
        
    otherwise
        xyw=[];
end




pointset_stats=[
    22   96 6.73e-16    0    0
    25  120 7.55e-15    6    0
    ];




