
function [xyw_bar,pointset_stats]=set_papanicolopulos_A_IV_barycentric(deg)

%--------------------------------------------------------------------
% CUBATURE RULE ON THE SIMPLEX [0 0; 0 1; 1 0] WITH ADE EQUAL TO deg.
% THE CUBATURE RULE IS DESCRIBED IN BARYCENTRICAL FORM.
%--------------------------------------------------------------------
% INPUT:
% deg: algebraic degree of precision (ADE) of the rule.
%--------------------------------------------------------------------
% OUTPUT:
% xyw_bar: MATRIX OF NODES xyw(:,1:3) IN BARYCENTRICAL COOR. AND WEIGHTS xyw(:,4).
%      REMARK: The sum of the weights makes 1/2.
%
% pointset_stats:
%    statistics of the pointset stored in a matrix. Each rows contains
%    degree, cardinality of the full pointset, moment error w.r.t.
%    orthn. Dubiner basis, number of points with negative weights,
%    number of points outside the domain.
%--------------------------------------------------------------------
% SOURCE:
% S-A. Papanicoloupolos
% "New fully symmetric and rotationally symmetric cubature rules on the
% triangle using minimal orthonormal bases".
% JCAM 294 (2016), 39-48.
% ** Using file rotationalsymmetry.txt from archives.
%--------------------------------------------------------------------
% REMARK:
% Rules are not appropriate for cubature when deg <= 16.
%--------------------------------------------------------------------
% | DEG | CARD | W- | PT.O|   M.E.   |Q.|
%--------------------------------------------------------------------
% |  10 |  24 |   0 |   0 | 3.16e-08 |PI|
% |  11 |  27 |   0 |   0 | 3.16e-08 |PI|
% |  12 |  31 |   0 |   3 | 3.31e-07 |P0|
% |  13 |  36 |   0 |   0 | 3.31e-07 |PI|
% |  14 |  40 |   0 |   3 | 4.97e-07 |P0|
% |  15 |  46 |   0 |   0 | 4.97e-07 |PI|
% |  16 |  51 |   0 |   6 | 2.31e-06 |P0|
% |  17 |  57 |   0 |   0 | 5.83e-16 |PI|
% |  18 |  64 |   1 |   0 | 4.86e-16 |NI|
% |  19 |  70 |   0 |   0 | 4.16e-16 |PI|
% |  20 |  78 |   0 |   0 | 4.89e-16 |PI|
% |  21 |  85 |   0 |   0 | 4.64e-16 |PI|
% |  22 |  93 |   0 |   0 | 7.56e-16 |PI|
% |  23 | 100 |   0 |   0 | 6.66e-16 |PI|
% |  24 | 109 |   0 |   0 | 7.81e-16 |PI|
% |  25 | 117 |   0 |   0 | 5.41e-16 |PI|
%--------------------------------------------------------------------
% LEGEND:
%--------------------------------------------------------------------
% DEG  : algebraic degree of precision
% CARD : cardinality of the pointset
% W-   : negative weights
% PT.O : points outside the simplex [0 0; 1 0; 0 1].
% M.E. : maximum moment error in inf norm.
% Q.: quality of the rule.
%        PI: pos. weights, points in simplex
%        NI: some neg. weights, points in simplex
%        P0: pos. weights, some points out of the simplex
%        N0: some neg. weights, some points out of the simplex
%--------------------------------------------------------------------


switch deg
    
    case  10
        
        % ALG. DEG.:   10
        % PTS CARD.:   24
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.2e-08
        
        xyw_bar=[
            1.45834576989294056881263372815738e-01 1.13985711600973777662959562917422e-02 8.42766851850608622598315378127154e-01 1.47111431252650217554878153691789e-02
            1.13985711600973777662959562917422e-02 8.42766851850608622598315378127154e-01 1.45834576989294056881263372815738e-01 1.47111431252650217554878153691789e-02
            8.42766851850608622598315378127154e-01 1.45834576989294056881263372815738e-01 1.13985711600973777662959562917422e-02 1.47111431252650217554878153691789e-02
            4.80639357548164503075582842939184e-01 3.96437069944491002337905172225874e-03 5.15396271752390666698318000271684e-01 1.49285931506815943564658866193895e-02
            3.96437069944491002337905172225874e-03 5.15396271752390666698318000271684e-01 4.80639357548164503075582842939184e-01 1.49285931506815943564658866193895e-02
            5.15396271752390666698318000271684e-01 4.80639357548164503075582842939184e-01 3.96437069944491002337905172225874e-03 1.49285931506815943564658866193895e-02
            2.83843621491539814874482949846879e-03 2.35341707042696801632786218760884e-01 7.61819856742387724324316877755336e-01 1.49473235488743663029742592129878e-02
            2.35341707042696801632786218760884e-01 7.61819856742387724324316877755336e-01 2.83843621491539814874482949846879e-03 1.49473235488743663029742592129878e-02
            7.61819856742387724324316877755336e-01 2.83843621491539814874482949846879e-03 2.35341707042696801632786218760884e-01 1.49473235488743663029742592129878e-02
            2.73824560962257976259870417834463e-02 4.33992882388709033669726977677783e-02 9.29218255664903347579297587799374e-01 1.53570001726465212021910389239565e-02
            4.33992882388709033669726977677783e-02 9.29218255664903347579297587799374e-01 2.73824560962257976259870417834463e-02 1.53570001726465212021910389239565e-02
            9.29218255664903347579297587799374e-01 2.73824560962257976259870417834463e-02 4.33992882388709033669726977677783e-02 1.53570001726465212021910389239565e-02
            2.90369916692392482371332107504713e-01 5.94849897662165297274050601572526e-02 6.50145093541390939329005504987435e-01 5.14405019668115956466003524383268e-02
            5.94849897662165297274050601572526e-02 6.50145093541390939329005504987435e-01 2.90369916692392482371332107504713e-01 5.14405019668115956466003524383268e-02
            6.50145093541390939329005504987435e-01 2.90369916692392482371332107504713e-01 5.94849897662165297274050601572526e-02 5.14405019668115956466003524383268e-02
            1.02579104349627109327158791529655e-01 1.45523268555174417615560855665535e-01 7.51897627095198473057280352804810e-01 5.51873929177541314161636876178818e-02
            1.45523268555174417615560855665535e-01 7.51897627095198473057280352804810e-01 1.02579104349627109327158791529655e-01 5.51873929177541314161636876178818e-02
            7.51897627095198473057280352804810e-01 1.02579104349627109327158791529655e-01 1.45523268555174417615560855665535e-01 5.51873929177541314161636876178818e-02
            8.91387186398934494580004184172140e-02 3.68934745933899244096920710944687e-01 5.41926535426207278689503255009186e-01 7.17179807098225763350995975997648e-02
            3.68934745933899244096920710944687e-01 5.41926535426207278689503255009186e-01 8.91387186398934494580004184172140e-02 7.17179807098225763350995975997648e-02
            5.41926535426207278689503255009186e-01 8.91387186398934494580004184172140e-02 3.68934745933899244096920710944687e-01 7.17179807098225763350995975997648e-02
            2.89306189170967176238491447293200e-01 2.27196143114623660963502516096924e-01 4.83497667714409162798006036609877e-01 9.50433977414775199576979503035545e-02
            2.27196143114623660963502516096924e-01 4.83497667714409162798006036609877e-01 2.89306189170967176238491447293200e-01 9.50433977414775199576979503035545e-02
            4.83497667714409162798006036609877e-01 2.89306189170967176238491447293200e-01 2.27196143114623660963502516096924e-01 9.50433977414775199576979503035545e-02
            ];
        
    case  11
        
        % ALG. DEG.:   11
        % PTS CARD.:   27
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.2e-08
        
        xyw_bar=[
            3.17327225265938497877904467259214e-02 3.38153693417108380803171030493104e-02 9.34451908131695319070786354131997e-01 1.38072791558211914464404657110208e-02
            3.38153693417108380803171030493104e-02 9.34451908131695319070786354131997e-01 3.17327225265938497877904467259214e-02 1.38072791558211914464404657110208e-02
            9.34451908131695319070786354131997e-01 3.17327225265938497877904467259214e-02 3.38153693417108380803171030493104e-02 1.38072791558211914464404657110208e-02
            1.59837355915728590582958190680074e-01 2.47248303942128599963723445398500e-02 8.15437813690058632687396311666816e-01 2.17032352733579070314107184458408e-02
            2.47248303942128599963723445398500e-02 8.15437813690058632687396311666816e-01 1.59837355915728590582958190680074e-01 2.17032352733579070314107184458408e-02
            8.15437813690058632687396311666816e-01 1.59837355915728590582958190680074e-01 2.47248303942128599963723445398500e-02 2.17032352733579070314107184458408e-02
            2.95071643134682745635011258400482e-02 1.69403876141335951732713738238090e-01 8.01088959545195766764891232014634e-01 2.70341277522031589453099797992763e-02
            1.69403876141335951732713738238090e-01 8.01088959545195766764891232014634e-01 2.95071643134682745635011258400482e-02 2.70341277522031589453099797992763e-02
            8.01088959545195766764891232014634e-01 2.95071643134682745635011258400482e-02 1.69403876141335951732713738238090e-01 2.70341277522031589453099797992763e-02
            2.28225764965843264109324195487716e-02 3.82904113878300034201629387098365e-01 5.94273309625115597754074769909494e-01 2.73312416539357484757122307428290e-02
            3.82904113878300034201629387098365e-01 5.94273309625115597754074769909494e-01 2.28225764965843264109324195487716e-02 2.73312416539357484757122307428290e-02
            5.94273309625115597754074769909494e-01 2.28225764965843264109324195487716e-02 3.82904113878300034201629387098365e-01 2.73312416539357484757122307428290e-02
            3.60374166927902639390168815225479e-01 2.51549434823748474521210027887719e-02 6.14470889589722624180012644501403e-01 2.83362851670311624574516429220239e-02
            2.51549434823748474521210027887719e-02 6.14470889589722624180012644501403e-01 3.60374166927902639390168815225479e-01 2.83362851670311624574516429220239e-02
            6.14470889589722624180012644501403e-01 3.60374166927902639390168815225479e-01 2.51549434823748474521210027887719e-02 2.83362851670311624574516429220239e-02
            1.52649488567569491248576696307282e-01 1.30991017977501539304441280364699e-01 7.16359493454928886180255176441278e-01 4.85436671261325475290604458677990e-02
            1.30991017977501539304441280364699e-01 7.16359493454928886180255176441278e-01 1.52649488567569491248576696307282e-01 4.85436671261325475290604458677990e-02
            7.16359493454928886180255176441278e-01 1.52649488567569491248576696307282e-01 1.30991017977501539304441280364699e-01 4.85436671261325475290604458677990e-02
            3.48698923498339419246860870771343e-01 1.20982496644817527631232678686501e-01 5.30318579856843053121906450542156e-01 5.28872990887763669287657819495507e-02
            1.20982496644817527631232678686501e-01 5.30318579856843053121906450542156e-01 3.48698923498339419246860870771343e-01 5.28872990887763669287657819495507e-02
            5.30318579856843053121906450542156e-01 3.48698923498339419246860870771343e-01 1.20982496644817527631232678686501e-01 5.28872990887763669287657819495507e-02
            1.21263846601003227876347523306322e-01 3.19556006508081080763616910189739e-01 5.59180146890915663604459950875025e-01 5.51622538993044378430319341077848e-02
            3.19556006508081080763616910189739e-01 5.59180146890915663604459950875025e-01 1.21263846601003227876347523306322e-01 5.51622538993044378430319341077848e-02
            5.59180146890915663604459950875025e-01 1.21263846601003227876347523306322e-01 3.19556006508081080763616910189739e-01 5.51622538993044378430319341077848e-02
            2.79902470374426504129417025978910e-01 2.69674314114379864992798729872447e-01 4.50423215511193630877784244148643e-01 5.85279442167708913169477114024630e-02
            2.69674314114379864992798729872447e-01 4.50423215511193630877784244148643e-01 2.79902470374426504129417025978910e-01 5.85279442167708913169477114024630e-02
            4.50423215511193630877784244148643e-01 2.79902470374426504129417025978910e-01 2.69674314114379864992798729872447e-01 5.85279442167708913169477114024630e-02
            ];
        
    case  12
        
        % ALG. DEG.:   12
        % PTS CARD.:   31
        % NEG. W.  :    0
        % OUT PTS. :    3
        % M.E.INF. : 3.3e-07
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333425851918718763045e-01 2.14094257267984991455200827203953e-02
            3.01131958012075158503950689237172e-02 -9.47160626824977980309228087207885e-02 1.06460286688129035503891373082297e+00 8.36681787236886797881271760068955e-05
            -9.47160626824977980309228087207885e-02 1.06460286688129035503891373082297e+00 3.01131958012075158503950689237172e-02 8.36681787236886797881271760068955e-05
            1.06460286688129035503891373082297e+00 3.01131958012075158503950689237172e-02 -9.47160626824977980309228087207885e-02 8.36681787236886797881271760068955e-05
            3.28665303214459636471644898847444e-01 8.04564520926081237772731213908628e-03 6.63289051576279597988161640387261e-01 1.24136370679848451192839320356143e-02
            8.04564520926081237772731213908628e-03 6.63289051576279597988161640387261e-01 3.28665303214459636471644898847444e-01 1.24136370679848451192839320356143e-02
            6.63289051576279597988161640387261e-01 3.28665303214459636471644898847444e-01 8.04564520926081237772731213908628e-03 1.24136370679848451192839320356143e-02
            4.81249211528512951807634578926809e-02 9.23861926809377842495507593412185e-01 2.80131520377708831404106604168192e-02 1.62399456292167641835444413800360e-02
            9.23861926809377842495507593412185e-01 2.80131520377708831404106604168192e-02 4.81249211528512951807634578926809e-02 1.62399456292167641835444413800360e-02
            2.80131520377708831404106604168192e-02 4.81249211528512951807634578926809e-02 9.23861926809377842495507593412185e-01 1.62399456292167641835444413800360e-02
            1.42850841527364574856662216006953e-01 3.10192386927205833446752336612917e-02 8.26129919779914789756958271027543e-01 2.41940393249840506040282406274855e-02
            3.10192386927205833446752336612917e-02 8.26129919779914789756958271027543e-01 1.42850841527364574856662216006953e-01 2.41940393249840506040282406274855e-02
            8.26129919779914789756958271027543e-01 1.42850841527364574856662216006953e-01 3.10192386927205833446752336612917e-02 2.41940393249840506040282406274855e-02
            2.08191925148008077917083369356988e-01 7.64831246745194648362087264104048e-01 2.69768281067972459652537509100512e-02 2.69673655860056538102753620478325e-02
            7.64831246745194648362087264104048e-01 2.69768281067972459652537509100512e-02 2.08191925148008077917083369356988e-01 2.69673655860056538102753620478325e-02
            2.69768281067972459652537509100512e-02 2.08191925148008077917083369356988e-01 7.64831246745194648362087264104048e-01 2.69673655860056538102753620478325e-02
            4.25825822203965698165006870112848e-01 5.50395696311636761777208448620513e-01 2.37784814843975400577846812666394e-02 2.77744276572151234383500906233166e-02
            5.50395696311636761777208448620513e-01 2.37784814843975400577846812666394e-02 4.25825822203965698165006870112848e-01 2.77744276572151234383500906233166e-02
            2.37784814843975400577846812666394e-02 4.25825822203965698165006870112848e-01 5.50395696311636761777208448620513e-01 2.77744276572151234383500906233166e-02
            3.23868233593083310051241596738691e-01 7.98513414941436766580906692070130e-02 5.96280424912772999412879926239839e-01 4.70167784513457823192261741951370e-02
            7.98513414941436766580906692070130e-02 5.96280424912772999412879926239839e-01 3.23868233593083310051241596738691e-01 4.70167784513457823192261741951370e-02
            5.96280424912772999412879926239839e-01 3.23868233593083310051241596738691e-01 7.98513414941436766580906692070130e-02 4.70167784513457823192261741951370e-02
            1.54176405413344208561454706796212e-01 7.07647956311328152345652142685140e-01 1.38175638275327639092893150518648e-01 4.85934144058270886534778298937454e-02
            7.07647956311328152345652142685140e-01 1.38175638275327639092893150518648e-01 1.54176405413344208561454706796212e-01 4.85934144058270886534778298937454e-02
            1.38175638275327639092893150518648e-01 1.54176405413344208561454706796212e-01 7.07647956311328152345652142685140e-01 4.85934144058270886534778298937454e-02
            3.49383618039354804096774387289770e-01 5.24861100312631445596878165815724e-01 1.25755281648013750306347446894506e-01 5.85052807110508429899731197565416e-02
            5.24861100312631445596878165815724e-01 1.25755281648013750306347446894506e-01 3.49383618039354804096774387289770e-01 5.85052807110508429899731197565416e-02
            1.25755281648013750306347446894506e-01 3.49383618039354804096774387289770e-01 5.24861100312631445596878165815724e-01 5.85052807110508429899731197565416e-02
            2.99315867642758581812501006425009e-01 2.29249615293796282777805117802927e-01 4.71434517063445190920845107029891e-01 6.44083010787133286578765023477899e-02
            2.29249615293796282777805117802927e-01 4.71434517063445190920845107029891e-01 2.99315867642758581812501006425009e-01 6.44083010787133286578765023477899e-02
            4.71434517063445190920845107029891e-01 2.99315867642758581812501006425009e-01 2.29249615293796282777805117802927e-01 6.44083010787133286578765023477899e-02
            ];
        
    case  13
        
        % ALG. DEG.:   13
        % PTS CARD.:   36
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.3e-07
        
        xyw_bar=[
            2.60167761021924195952159664102510e-02 2.40222138933009303485288654655960e-02 9.49961010004506611892338696634397e-01 8.07997189224625260439704987902587e-03
            2.40222138933009303485288654655960e-02 9.49961010004506611892338696634397e-01 2.60167761021924195952159664102510e-02 8.07997189224625260439704987902587e-03
            9.49961010004506611892338696634397e-01 2.60167761021924195952159664102510e-02 2.40222138933009303485288654655960e-02 8.07997189224625260439704987902587e-03
            1.88836312986729974727762026986966e-02 1.22080831671691814044322654808639e-01 8.59035537029635198891241998353507e-01 1.28920291377734655413611619678704e-02
            1.22080831671691814044322654808639e-01 8.59035537029635198891241998353507e-01 1.88836312986729974727762026986966e-02 1.28920291377734655413611619678704e-02
            8.59035537029635198891241998353507e-01 1.88836312986729974727762026986966e-02 1.22080831671691814044322654808639e-01 1.28920291377734655413611619678704e-02
            1.31639171218805528518203118437668e-01 2.28747050990018990779262253454363e-02 8.45486123682192558526082848402439e-01 1.65602379220932578729463813260736e-02
            2.28747050990018990779262253454363e-02 8.45486123682192558526082848402439e-01 1.31639171218805528518203118437668e-01 1.65602379220932578729463813260736e-02
            8.45486123682192558526082848402439e-01 1.31639171218805528518203118437668e-01 2.28747050990018990779262253454363e-02 1.65602379220932578729463813260736e-02
            2.07315285531347315295391808831482e-02 2.80737796608389877928146916019614e-01 6.98530674838475418297889518726151e-01 1.90231358688970342285529113723896e-02
            2.80737796608389877928146916019614e-01 6.98530674838475418297889518726151e-01 2.07315285531347315295391808831482e-02 1.90231358688970342285529113723896e-02
            6.98530674838475418297889518726151e-01 2.07315285531347315295391808831482e-02 2.80737796608389877928146916019614e-01 1.90231358688970342285529113723896e-02
            4.76642895184469561620232980203582e-01 5.04849456195796686408527875755681e-01 1.85076486197337519712391440407373e-02 1.93596574803232522599127918283557e-02
            5.04849456195796686408527875755681e-01 1.85076486197337519712391440407373e-02 4.76642895184469561620232980203582e-01 1.93596574803232522599127918283557e-02
            1.85076486197337519712391440407373e-02 4.76642895184469561620232980203582e-01 5.04849456195796686408527875755681e-01 1.93596574803232522599127918283557e-02
            3.02386954450793832283039819230908e-01 1.99757527883214214270424236019608e-02 6.77637292760884801801068988424959e-01 1.95837983592238654151351795462688e-02
            1.99757527883214214270424236019608e-02 6.77637292760884801801068988424959e-01 3.02386954450793832283039819230908e-01 1.95837983592238654151351795462688e-02
            6.77637292760884801801068988424959e-01 3.02386954450793832283039819230908e-01 1.99757527883214214270424236019608e-02 1.95837983592238654151351795462688e-02
            1.01539779723998968563591915881261e-01 1.19028215975854986807824786865240e-01 7.79432004300146030750795489439042e-01 3.02814938259689976629562835341858e-02
            1.19028215975854986807824786865240e-01 7.79432004300146030750795489439042e-01 1.01539779723998968563591915881261e-01 3.02814938259689976629562835341858e-02
            7.79432004300146030750795489439042e-01 1.01539779723998968563591915881261e-01 1.19028215975854986807824786865240e-01 3.02814938259689976629562835341858e-02
            2.49614427797335225456265561660985e-01 1.03589966905477309788707884763426e-01 6.46795605297187492510602169204503e-01 3.81640664798614942521126636165718e-02
            1.03589966905477309788707884763426e-01 6.46795605297187492510602169204503e-01 2.49614427797335225456265561660985e-01 3.81640664798614942521126636165718e-02
            6.46795605297187492510602169204503e-01 2.49614427797335225456265561660985e-01 1.03589966905477309788707884763426e-01 3.81640664798614942521126636165718e-02
            1.02029240543062130197604631121067e-01 2.76888257363515577491597241532872e-01 6.21082502093422306188585935160518e-01 3.81993383171111006157971701213683e-02
            2.76888257363515577491597241532872e-01 6.21082502093422306188585935160518e-01 1.02029240543062130197604631121067e-01 3.81993383171111006157971701213683e-02
            6.21082502093422306188585935160518e-01 1.02029240543062130197604631121067e-01 2.76888257363515577491597241532872e-01 3.81993383171111006157971701213683e-02
            2.26789276022037022961796992603922e-01 2.33728374217554740743807428771106e-01 5.39482349760408208538819962996058e-01 4.08471528918510698447086326723365e-02
            2.33728374217554740743807428771106e-01 5.39482349760408208538819962996058e-01 2.26789276022037022961796992603922e-01 4.08471528918510698447086326723365e-02
            5.39482349760408208538819962996058e-01 2.26789276022037022961796992603922e-01 2.33728374217554740743807428771106e-01 4.08471528918510698447086326723365e-02
            4.37409517911078460361551378809963e-01 9.59920933203395609911723340701428e-02 4.66598388768582061914003134006634e-01 4.10856959865649046248492481936410e-02
            9.59920933203395609911723340701428e-02 4.66598388768582061914003134006634e-01 4.37409517911078460361551378809963e-01 4.10856959865649046248492481936410e-02
            4.66598388768582061914003134006634e-01 4.37409517911078460361551378809963e-01 9.59920933203395609911723340701428e-02 4.10856959865649046248492481936410e-02
            3.80758579162244537030801438959315e-01 2.27855357118668860882948479229526e-01 3.91386063719086574330674466182245e-01 4.92567551714187049083371050528513e-02
            2.27855357118668860882948479229526e-01 3.91386063719086574330674466182245e-01 3.80758579162244537030801438959315e-01 4.92567551714187049083371050528513e-02
            3.91386063719086574330674466182245e-01 3.80758579162244537030801438959315e-01 2.27855357118668860882948479229526e-01 4.92567551714187049083371050528513e-02
            ];
        
    case  14
        
        % ALG. DEG.:   14
        % PTS CARD.:   40
        % NEG. W.  :    0
        % OUT PTS. :    3
        % M.E.INF. : 5.0e-07
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333425851918718763045e-01 1.72196019973308092620989384613495e-02
            1.22379515529846980004258227836544e-01 -2.21164638633246993237868593951134e-02 8.99736948333477726258422535465797e-01 1.22343703244302934481091416074605e-03
            -2.21164638633246993237868593951134e-02 8.99736948333477726258422535465797e-01 1.22379515529846980004258227836544e-01 1.22343703244302934481091416074605e-03
            8.99736948333477726258422535465797e-01 1.22379515529846980004258227836544e-01 -2.21164638633246993237868593951134e-02 1.22343703244302934481091416074605e-03
            2.56919929576503835999545799495536e-01 7.42961945176951910063678496953798e-01 1.18125246544198425624472292838618e-04 5.62580606530807027232388151105624e-03
            7.42961945176951910063678496953798e-01 1.18125246544198425624472292838618e-04 2.56919929576503835999545799495536e-01 5.62580606530807027232388151105624e-03
            1.18125246544198425624472292838618e-04 2.56919929576503835999545799495536e-01 7.42961945176951910063678496953798e-01 5.62580606530807027232388151105624e-03
            2.24126981487152709959786989202257e-02 2.12963387678650611922748225879332e-02 9.56290963083419653933958670677384e-01 6.18034487246762326284610367110872e-03
            2.12963387678650611922748225879332e-02 9.56290963083419653933958670677384e-01 2.24126981487152709959786989202257e-02 6.18034487246762326284610367110872e-03
            9.56290963083419653933958670677384e-01 2.24126981487152709959786989202257e-02 2.12963387678650611922748225879332e-02 6.18034487246762326284610367110872e-03
            1.07983615865477042272679852885631e-01 8.73605481594919575094593255926156e-01 1.84109025396034242660903146315832e-02 1.09576973579726157093094940364608e-02
            8.73605481594919575094593255926156e-01 1.84109025396034242660903146315832e-02 1.07983615865477042272679852885631e-01 1.09576973579726157093094940364608e-02
            1.84109025396034242660903146315832e-02 1.07983615865477042272679852885631e-01 8.73605481594919575094593255926156e-01 1.09576973579726157093094940364608e-02
            4.54399148709708655680117317388067e-01 8.40211889048260093804110226756166e-03 5.37198732399808753790182436205214e-01 1.14623430106397840300136437008405e-02
            8.40211889048260093804110226756166e-03 5.37198732399808753790182436205214e-01 4.54399148709708655680117317388067e-01 1.14623430106397840300136437008405e-02
            5.37198732399808753790182436205214e-01 4.54399148709708655680117317388067e-01 8.40211889048260093804110226756166e-03 1.14623430106397840300136437008405e-02
            2.60936998850153534235118968354072e-01 2.57352520882206184382745561833872e-02 7.13327749061625882021075994998682e-01 2.22957840398664147718577055456990e-02
            2.57352520882206184382745561833872e-02 7.13327749061625882021075994998682e-01 2.60936998850153534235118968354072e-01 2.22957840398664147718577055456990e-02
            7.13327749061625882021075994998682e-01 2.60936998850153534235118968354072e-01 2.57352520882206184382745561833872e-02 2.22957840398664147718577055456990e-02
            1.08651813403198613250388859796658e-01 4.73179795037884515229720250317769e-02 8.44030207093012907471063499542652e-01 2.33828631874602073381641531568675e-02
            4.73179795037884515229720250317769e-02 8.44030207093012907471063499542652e-01 1.08651813403198613250388859796658e-01 2.33828631874602073381641531568675e-02
            8.44030207093012907471063499542652e-01 1.08651813403198613250388859796658e-01 4.73179795037884515229720250317769e-02 2.33828631874602073381641531568675e-02
            3.63151184998712905915851933968952e-01 6.00340345913397799293420575850178e-01 3.65084690878892947907274901808705e-02 2.65580672951120852487427725918678e-02
            6.00340345913397799293420575850178e-01 3.65084690878892947907274901808705e-02 3.63151184998712905915851933968952e-01 2.65580672951120852487427725918678e-02
            3.65084690878892947907274901808705e-02 3.63151184998712905915851933968952e-01 6.00340345913397799293420575850178e-01 2.65580672951120852487427725918678e-02
            1.88070088519434769347427049979160e-01 7.41933521291306963263423313037492e-01 6.99963901892582951447252526122611e-02 3.08504882707558922283475055792223e-02
            7.41933521291306963263423313037492e-01 6.99963901892582951447252526122611e-02 1.88070088519434769347427049979160e-01 3.08504882707558922283475055792223e-02
            6.99963901892582951447252526122611e-02 1.88070088519434769347427049979160e-01 7.41933521291306963263423313037492e-01 3.08504882707558922283475055792223e-02
            3.93963047077244832294695697783027e-01 8.32192413656172996772042438351491e-02 5.22817711557137854150312250567367e-01 4.06810514485703364484514565901918e-02
            8.32192413656172996772042438351491e-02 5.22817711557137854150312250567367e-01 3.93963047077244832294695697783027e-01 4.06810514485703364484514565901918e-02
            5.22817711557137854150312250567367e-01 3.93963047077244832294695697783027e-01 8.32192413656172996772042438351491e-02 4.06810514485703364484514565901918e-02
            2.04286427343950721668974779277050e-01 1.37280339301185688727358069627371e-01 6.58433233354863589603667151095578e-01 4.36523977472310989456083518689411e-02
            1.37280339301185688727358069627371e-01 6.58433233354863589603667151095578e-01 2.04286427343950721668974779277050e-01 4.36523977472310989456083518689411e-02
            6.58433233354863589603667151095578e-01 2.04286427343950721668974779277050e-01 1.37280339301185688727358069627371e-01 4.36523977472310989456083518689411e-02
            3.15511922686269707938322426343802e-01 5.33376567358602837742864721803926e-01 1.51111509955127454318812851852272e-01 5.08937432805527806278789171301469e-02
            5.33376567358602837742864721803926e-01 1.51111509955127454318812851852272e-01 3.15511922686269707938322426343802e-01 5.08937432805527806278789171301469e-02
            1.51111509955127454318812851852272e-01 3.15511922686269707938322426343802e-01 5.33376567358602837742864721803926e-01 5.08937432805527806278789171301469e-02
            3.24023300923112766991351918477449e-01 2.28102726523556703641659737513692e-01 4.47873972553330501611412728379946e-01 5.38294423925099324579335302587424e-02
            2.28102726523556703641659737513692e-01 4.47873972553330501611412728379946e-01 3.24023300923112766991351918477449e-01 5.38294423925099324579335302587424e-02
            4.47873972553330501611412728379946e-01 3.24023300923112766991351918477449e-01 2.28102726523556703641659737513692e-01 5.38294423925099324579335302587424e-02
            ];
        
    case  15
        
        % ALG. DEG.:   15
        % PTS CARD.:   46
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.0e-07
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333425851918718763045e-01 4.67496120877831169981853065564792e-02
            1.84347954761566144854523230378618e-02 2.10205785689865885801363987184232e-02 9.60544625954856901017819836852141e-01 5.02785609724814685356220067546928e-03
            2.10205785689865885801363987184232e-02 9.60544625954856901017819836852141e-01 1.84347954761566144854523230378618e-02 5.02785609724814685356220067546928e-03
            9.60544625954856901017819836852141e-01 1.84347954761566144854523230378618e-02 2.10205785689865885801363987184232e-02 5.02785609724814685356220067546928e-03
            9.44693915416657953310064499419241e-02 1.40084921643385436340611960304159e-02 8.91522116293995692259954921610188e-01 7.59868194010495084239620311450381e-03
            1.40084921643385436340611960304159e-02 8.91522116293995692259954921610188e-01 9.44693915416657953310064499419241e-02 7.59868194010495084239620311450381e-03
            8.91522116293995692259954921610188e-01 9.44693915416657953310064499419241e-02 1.40084921643385436340611960304159e-02 7.59868194010495084239620311450381e-03
            2.21059537257312810343279352309764e-01 1.46001571927490553237660719787527e-02 7.64340305549938148210742383525940e-01 1.07321195836478366913979698438197e-02
            1.46001571927490553237660719787527e-02 7.64340305549938148210742383525940e-01 2.21059537257312810343279352309764e-01 1.07321195836478366913979698438197e-02
            7.64340305549938148210742383525940e-01 2.21059537257312810343279352309764e-01 1.46001571927490553237660719787527e-02 1.07321195836478366913979698438197e-02
            1.83161255802240172108064086842205e-02 1.06831432538477322236047939441050e-01 8.74852441881298625858676132338587e-01 1.08308519657447071410372174682379e-02
            1.06831432538477322236047939441050e-01 8.74852441881298625858676132338587e-01 1.83161255802240172108064086842205e-02 1.08308519657447071410372174682379e-02
            8.74852441881298625858676132338587e-01 1.83161255802240172108064086842205e-02 1.06831432538477322236047939441050e-01 1.08308519657447071410372174682379e-02
            3.84111820811406545761457209664513e-01 1.35065653189919161053511587056164e-02 6.02381613869601539867915107606677e-01 1.22487272504865716410415998893768e-02
            1.35065653189919161053511587056164e-02 6.02381613869601539867915107606677e-01 3.84111820811406545761457209664513e-01 1.22487272504865716410415998893768e-02
            6.02381613869601539867915107606677e-01 3.84111820811406545761457209664513e-01 1.35065653189919161053511587056164e-02 1.22487272504865716410415998893768e-02
            1.76095767990848382777979708180283e-02 2.46869440424026448388516996601538e-01 7.35520982776888709864238080626819e-01 1.43978395837907422599721840583697e-02
            2.46869440424026448388516996601538e-01 7.35520982776888709864238080626819e-01 1.76095767990848382777979708180283e-02 1.43978395837907422599721840583697e-02
            7.35520982776888709864238080626819e-01 1.76095767990848382777979708180283e-02 2.46869440424026448388516996601538e-01 1.43978395837907422599721840583697e-02
            1.62111042661076745763359951979510e-02 4.20027323435867572243296308442950e-01 5.63761572298024704608110369008500e-01 1.49838071522019559284055745251862e-02
            4.20027323435867572243296308442950e-01 5.63761572298024704608110369008500e-01 1.62111042661076745763359951979510e-02 1.49838071522019559284055745251862e-02
            5.63761572298024704608110369008500e-01 1.62111042661076745763359951979510e-02 4.20027323435867572243296308442950e-01 1.49838071522019559284055745251862e-02
            9.45835225979976129684345664827561e-02 7.81937852376641401308532408620522e-02 8.27222692164338191389560961397365e-01 1.92326833970584594457609028950174e-02
            7.81937852376641401308532408620522e-02 8.27222692164338191389560961397365e-01 9.45835225979976129684345664827561e-02 1.92326833970584594457609028950174e-02
            8.27222692164338191389560961397365e-01 9.45835225979976129684345664827561e-02 7.81937852376641401308532408620522e-02 1.92326833970584594457609028950174e-02
            2.23020541186686460655153041443555e-01 6.90246522127956757541866750216286e-02 7.07954806600517905224023706978187e-01 2.08346848826061990800351253483313e-02
            6.90246522127956757541866750216286e-02 7.07954806600517905224023706978187e-01 2.23020541186686460655153041443555e-01 2.08346848826061990800351253483313e-02
            7.07954806600517905224023706978187e-01 2.23020541186686460655153041443555e-01 6.90246522127956757541866750216286e-02 2.08346848826061990800351253483313e-02
            9.13686335566708390798495997842110e-02 1.97441620058397909742708975500136e-01 7.11189746384931265055229232530110e-01 2.85769656783392486665285048275109e-02
            1.97441620058397909742708975500136e-01 7.11189746384931265055229232530110e-01 9.13686335566708390798495997842110e-02 2.85769656783392486665285048275109e-02
            7.11189746384931265055229232530110e-01 9.13686335566708390798495997842110e-02 1.97441620058397909742708975500136e-01 2.85769656783392486665285048275109e-02
            3.84530936051927341168266138993204e-01 7.30977621267044908037746608897578e-02 5.42371301821368168027959200117039e-01 2.90316778855890765542735465487567e-02
            7.30977621267044908037746608897578e-02 5.42371301821368168027959200117039e-01 3.84530936051927341168266138993204e-01 2.90316778855890765542735465487567e-02
            5.42371301821368168027959200117039e-01 3.84530936051927341168266138993204e-01 7.30977621267044908037746608897578e-02 2.90316778855890765542735465487567e-02
            2.15038741798252275305713965281029e-01 1.55893061854053271986231266055256e-01 6.29068196347694508219205999921542e-01 3.02402832704538639119196119509070e-02
            1.55893061854053271986231266055256e-01 6.29068196347694508219205999921542e-01 2.15038741798252275305713965281029e-01 3.02402832704538639119196119509070e-02
            6.29068196347694508219205999921542e-01 2.15038741798252275305713965281029e-01 1.55893061854053271986231266055256e-01 3.02402832704538639119196119509070e-02
            8.53189701817285567342352692321583e-02 3.59007553448066507595370921990252e-01 5.55673476370204921792606000963133e-01 3.25115218538971559314454395916982e-02
            3.59007553448066507595370921990252e-01 5.55673476370204921792606000963133e-01 8.53189701817285567342352692321583e-02 3.25115218538971559314454395916982e-02
            5.55673476370204921792606000963133e-01 8.53189701817285567342352692321583e-02 3.59007553448066507595370921990252e-01 3.25115218538971559314454395916982e-02
            2.04230412205111150347747184241598e-01 2.86157214943028759268628391509992e-01 5.09612372851860007116897577361669e-01 4.01871265636518232300389286137943e-02
            2.86157214943028759268628391509992e-01 5.09612372851860007116897577361669e-01 2.04230412205111150347747184241598e-01 4.01871265636518232300389286137943e-02
            5.09612372851860007116897577361669e-01 2.04230412205111150347747184241598e-01 2.86157214943028759268628391509992e-01 4.01871265636518232300389286137943e-02
            3.63890898963924347420118010631995e-01 1.81085973861799420792095816068468e-01 4.55023127174276231787786173299537e-01 4.13153021992516272797857368459518e-02
            1.81085973861799420792095816068468e-01 4.55023127174276231787786173299537e-01 3.63890898963924347420118010631995e-01 4.13153021992516272797857368459518e-02
            4.55023127174276231787786173299537e-01 3.63890898963924347420118010631995e-01 1.81085973861799420792095816068468e-01 4.13153021992516272797857368459518e-02
            ];
        
    case  16
        
        % ALG. DEG.:   16
        % PTS CARD.:   51
        % NEG. W.  :    0
        % OUT PTS. :    6
        % M.E.INF. : 2.3e-06
        
        xyw_bar=[
            2.60234434945475379130641613301123e-01 -7.79607539117141944140509934868533e-03 7.47561640445696062862168673746055e-01 2.65840425585919934090006755411650e-03
            -7.79607539117141944140509934868533e-03 7.47561640445696062862168673746055e-01 2.60234434945475379130641613301123e-01 2.65840425585919934090006755411650e-03
            7.47561640445696062862168673746055e-01 2.60234434945475379130641613301123e-01 -7.79607539117141944140509934868533e-03 2.65840425585919934090006755411650e-03
            8.76544356764728666009389712598932e-02 9.09443871530899672350756191008259e-01 2.90169279262741941494141428847797e-03 3.39962230325435103145403914481903e-03
            9.09443871530899672350756191008259e-01 2.90169279262741941494141428847797e-03 8.76544356764728666009389712598932e-02 3.39962230325435103145403914481903e-03
            2.90169279262741941494141428847797e-03 8.76544356764728666009389712598932e-02 9.09443871530899672350756191008259e-01 3.39962230325435103145403914481903e-03
            3.00673047577842300626116411876865e-01 7.00388097083945782017622150306124e-01 -1.06114466178808264373856218298897e-03 3.51910900349891760108267391160553e-03
            7.00388097083945782017622150306124e-01 -1.06114466178808264373856218298897e-03 3.00673047577842300626116411876865e-01 3.51910900349891760108267391160553e-03
            -1.06114466178808264373856218298897e-03 3.00673047577842300626116411876865e-01 7.00388097083945782017622150306124e-01 3.51910900349891760108267391160553e-03
            1.88499253609186803726238679246308e-02 1.64729175426546053939702574098192e-02 9.64677157096426762805663202016149e-01 4.04414041170866010721507421976639e-03
            1.64729175426546053939702574098192e-02 9.64677157096426762805663202016149e-01 1.88499253609186803726238679246308e-02 4.04414041170866010721507421976639e-03
            9.64677157096426762805663202016149e-01 1.88499253609186803726238679246308e-02 1.64729175426546053939702574098192e-02 4.04414041170866010721507421976639e-03
            9.84166842500529825965571717460989e-02 1.19410067117379486284489686909183e-02 8.89642309038209067040270383586176e-01 6.82947115125906532157040729202890e-03
            1.19410067117379486284489686909183e-02 8.89642309038209067040270383586176e-01 9.84166842500529825965571717460989e-02 6.82947115125906532157040729202890e-03
            8.89642309038209067040270383586176e-01 9.84166842500529825965571717460989e-02 1.19410067117379486284489686909183e-02 6.82947115125906532157040729202890e-03
            4.60403016961182465216495529602980e-01 1.17879693492858039771675748852431e-02 5.27809013689531703050761279882863e-01 1.04401675585101185589342875914554e-02
            1.17879693492858039771675748852431e-02 5.27809013689531703050761279882863e-01 4.60403016961182465216495529602980e-01 1.04401675585101185589342875914554e-02
            5.27809013689531703050761279882863e-01 4.60403016961182465216495529602980e-01 1.17879693492858039771675748852431e-02 1.04401675585101185589342875914554e-02
            7.38299110030455618680633733674767e-02 8.68681210960493666028980896953726e-01 5.74888780364607443473801140498836e-02 1.47307124919196109208430200965267e-02
            8.68681210960493666028980896953726e-01 5.74888780364607443473801140498836e-02 7.38299110030455618680633733674767e-02 1.47307124919196109208430200965267e-02
            5.74888780364607443473801140498836e-02 7.38299110030455618680633733674767e-02 8.68681210960493666028980896953726e-01 1.47307124919196109208430200965267e-02
            1.86109933337395866415064915599942e-01 7.87962853056884626745670630043605e-01 2.59272136057194790836888387275394e-02 1.49971889894492862310704239803272e-02
            7.87962853056884626745670630043605e-01 2.59272136057194790836888387275394e-02 1.86109933337395866415064915599942e-01 1.49971889894492862310704239803272e-02
            2.59272136057194790836888387275394e-02 1.86109933337395866415064915599942e-01 7.87962853056884626745670630043605e-01 1.49971889894492862310704239803272e-02
            3.83202318286473053721152837169939e-01 5.88557349843886412621429826685926e-01 2.82403318696405891685685674019624e-02 1.65632144925227167697734387274977e-02
            5.88557349843886412621429826685926e-01 2.82403318696405891685685674019624e-02 3.83202318286473053721152837169939e-01 1.65632144925227167697734387274977e-02
            2.82403318696405891685685674019624e-02 3.83202318286473053721152837169939e-01 5.88557349843886412621429826685926e-01 1.65632144925227167697734387274977e-02
            1.86107887529837312534652937756618e-01 4.21293595759998432126458567381633e-02 7.71762752894162851191595109412447e-01 1.97481958439655394532685761532775e-02
            4.21293595759998432126458567381633e-02 7.71762752894162851191595109412447e-01 1.86107887529837312534652937756618e-01 1.97481958439655394532685761532775e-02
            7.71762752894162851191595109412447e-01 1.86107887529837312534652937756618e-01 4.21293595759998432126458567381633e-02 1.97481958439655394532685761532775e-02
            3.29435670084687715064575286305626e-01 3.83980593573524930928186904566246e-02 6.32166270557959819598181638866663e-01 2.12031289189748527179713022405849e-02
            3.83980593573524930928186904566246e-02 6.32166270557959819598181638866663e-01 3.29435670084687715064575286305626e-01 2.12031289189748527179713022405849e-02
            6.32166270557959819598181638866663e-01 3.29435670084687715064575286305626e-01 3.83980593573524930928186904566246e-02 2.12031289189748527179713022405849e-02
            2.72057325895406876270499196834862e-01 6.51051709131074241021508441917831e-01 7.68909649735188827079923612473067e-02 2.44486452221181309318875918279446e-02
            6.51051709131074241021508441917831e-01 7.68909649735188827079923612473067e-02 2.72057325895406876270499196834862e-01 2.44486452221181309318875918279446e-02
            7.68909649735188827079923612473067e-02 2.72057325895406876270499196834862e-01 6.51051709131074241021508441917831e-01 2.44486452221181309318875918279446e-02
            1.41235380655937520222309444761777e-01 7.29399317013244119500825490831630e-01 1.29365302330818332521289448777679e-01 2.81498697514387455353990219464322e-02
            7.29399317013244119500825490831630e-01 1.29365302330818332521289448777679e-01 1.41235380655937520222309444761777e-01 2.81498697514387455353990219464322e-02
            1.29365302330818332521289448777679e-01 1.41235380655937520222309444761777e-01 7.29399317013244119500825490831630e-01 2.81498697514387455353990219464322e-02
            4.49015635136169932728478215722134e-01 9.76218419601917047900840884722129e-02 4.53362522903638376359225503620110e-01 3.69021164809547261298483533664694e-02
            9.76218419601917047900840884722129e-02 4.53362522903638376359225503620110e-01 4.49015635136169932728478215722134e-01 3.69021164809547261298483533664694e-02
            4.53362522903638376359225503620110e-01 4.49015635136169932728478215722134e-01 9.76218419601917047900840884722129e-02 3.69021164809547261298483533664694e-02
            2.66798367513168654951982716738712e-01 1.37399270890935149003198034733941e-01 5.95802361595896168289243632898433e-01 3.70477362032751514342443499572255e-02
            1.37399270890935149003198034733941e-01 5.95802361595896168289243632898433e-01 2.66798367513168654951982716738712e-01 3.70477362032751514342443499572255e-02
            5.95802361595896168289243632898433e-01 2.66798367513168654951982716738712e-01 1.37399270890935149003198034733941e-01 3.70477362032751514342443499572255e-02
            2.82382457106112261246977368500666e-01 5.34917330044390570975565424305387e-01 1.82700212849497223288608438451774e-01 4.13325329790161025145245332623745e-02
            5.34917330044390570975565424305387e-01 1.82700212849497223288608438451774e-01 2.82382457106112261246977368500666e-01 4.13325329790161025145245332623745e-02
            1.82700212849497223288608438451774e-01 2.82382457106112261246977368500666e-01 5.34917330044390570975565424305387e-01 4.13325329790161025145245332623745e-02
            3.49747321380104725374593499509501e-01 2.38132150850675849973470121767605e-01 4.12120527769219369140785147465067e-01 4.73190772756081554084595097720012e-02
            2.38132150850675849973470121767605e-01 4.12120527769219369140785147465067e-01 3.49747321380104725374593499509501e-01 4.73190772756081554084595097720012e-02
            4.12120527769219369140785147465067e-01 3.49747321380104725374593499509501e-01 2.38132150850675849973470121767605e-01 4.73190772756081554084595097720012e-02
            ];
        
    case  17
        
        % ALG. DEG.:   17
        % PTS CARD.:   57
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.8e-16
        
        xyw_bar=[
            1.47209661956265915744301864265253e-02 1.76399848555067323419009994722728e-02 9.67639048948866653532263626402710e-01 3.37703600809952965422500170689091e-03
            1.76399848555067323419009994722728e-02 9.67639048948866653532263626402710e-01 1.47209661956265915744301864265253e-02 3.37703600809952965422500170689091e-03
            9.67639048948866653532263626402710e-01 1.47209661956265915744301864265253e-02 1.76399848555067323419009994722728e-02 3.37703600809952965422500170689091e-03
            7.58278723376997892957618319087487e-02 7.69747584377225568808000844001072e-03 9.16474651818527941138370351836784e-01 3.81992567019393035668284319683607e-03
            7.69747584377225568808000844001072e-03 9.16474651818527941138370351836784e-01 7.58278723376997892957618319087487e-02 3.81992567019393035668284319683607e-03
            9.16474651818527941138370351836784e-01 7.58278723376997892957618319087487e-02 7.69747584377225568808000844001072e-03 3.81992567019393035668284319683607e-03
            3.10585516558464003189499180734856e-01 8.32009195104604741544118695628640e-03 6.81094391490490003171487387589877e-01 6.62402608144134919659906657329884e-03
            8.32009195104604741544118695628640e-03 6.81094391490490003171487387589877e-01 3.10585516558464003189499180734856e-01 6.62402608144134919659906657329884e-03
            6.81094391490490003171487387589877e-01 3.10585516558464003189499180734856e-01 8.32009195104604741544118695628640e-03 6.62402608144134919659906657329884e-03
            1.44883146249404001770777483670827e-02 9.01640395800713256857505939478870e-02 8.95347645794988267198277753777802e-01 7.28490073487348104713801077991775e-03
            9.01640395800713256857505939478870e-02 8.95347645794988267198277753777802e-01 1.44883146249404001770777483670827e-02 7.28490073487348104713801077991775e-03
            8.95347645794988267198277753777802e-01 1.44883146249404001770777483670827e-02 9.01640395800713256857505939478870e-02 7.28490073487348104713801077991775e-03
            1.78143852253019957521118499244039e-01 1.64238580465156137899729316131925e-02 8.05432289700464432158355521096382e-01 9.96500247340540747098547313953532e-03
            1.64238580465156137899729316131925e-02 8.05432289700464432158355521096382e-01 1.78143852253019957521118499244039e-01 9.96500247340540747098547313953532e-03
            8.05432289700464432158355521096382e-01 1.78143852253019957521118499244039e-01 1.64238580465156137899729316131925e-02 9.96500247340540747098547313953532e-03
            4.66133550795934714638946161358035e-01 1.23355817687377852104324560400528e-02 5.21530867435327394332489348016679e-01 1.02763450985959500783017972480593e-02
            1.23355817687377852104324560400528e-02 5.21530867435327394332489348016679e-01 4.66133550795934714638946161358035e-01 1.02763450985959500783017972480593e-02
            5.21530867435327394332489348016679e-01 4.66133550795934714638946161358035e-01 1.23355817687377852104324560400528e-02 1.02763450985959500783017972480593e-02
            1.47365661662822756972079929482788e-02 2.09325171157457634585341565980343e-01 7.75938262676260093186897393024992e-01 1.03094913037771809816156576289359e-02
            2.09325171157457634585341565980343e-01 7.75938262676260093186897393024992e-01 1.47365661662822756972079929482788e-02 1.03094913037771809816156576289359e-02
            7.75938262676260093186897393024992e-01 1.47365661662822756972079929482788e-02 2.09325171157457634585341565980343e-01 1.03094913037771809816156576289359e-02
            1.43325355478803179165581127563200e-02 3.58602315795423676192399398132693e-01 6.27065148656696047524405912554357e-01 1.16022291368002225531164128824457e-02
            3.58602315795423676192399398132693e-01 6.27065148656696047524405912554357e-01 1.43325355478803179165581127563200e-02 1.16022291368002225531164128824457e-02
            6.27065148656696047524405912554357e-01 1.43325355478803179165581127563200e-02 3.58602315795423676192399398132693e-01 1.16022291368002225531164128824457e-02
            7.53033493906421630414982359980058e-02 5.59681329947420772019661683316372e-02 8.68728517614615669550914844876388e-01 1.24697437874820918346507880869467e-02
            5.59681329947420772019661683316372e-02 8.68728517614615669550914844876388e-01 7.53033493906421630414982359980058e-02 1.24697437874820918346507880869467e-02
            8.68728517614615669550914844876388e-01 7.53033493906421630414982359980058e-02 5.59681329947420772019661683316372e-02 1.24697437874820918346507880869467e-02
            3.17339393501727851099047938987496e-01 5.16948784187994947458832939446438e-02 6.30965728079472598643917535810033e-01 1.96324795155319517525605022001400e-02
            5.16948784187994947458832939446438e-02 6.30965728079472598643917535810033e-01 3.17339393501727851099047938987496e-01 1.96324795155319517525605022001400e-02
            6.30965728079472598643917535810033e-01 3.17339393501727851099047938987496e-01 5.16948784187994947458832939446438e-02 1.96324795155319517525605022001400e-02
            7.53542654903545661149166789982701e-02 1.54598211665730095321080739267927e-01 7.70047522843915421830729428620543e-01 2.01015070751451897790307299374035e-02
            1.54598211665730095321080739267927e-01 7.70047522843915421830729428620543e-01 7.53542654903545661149166789982701e-02 2.01015070751451897790307299374035e-02
            7.70047522843915421830729428620543e-01 7.53542654903545661149166789982701e-02 1.54598211665730095321080739267927e-01 2.01015070751451897790307299374035e-02
            1.82570774440876770761121861141874e-01 2.66530059635146754359880105766933e-01 5.50899165923976474878998033091193e-01 2.09060455685651375157263487380987e-02
            2.66530059635146754359880105766933e-01 5.50899165923976474878998033091193e-01 1.82570774440876770761121861141874e-01 2.09060455685651375157263487380987e-02
            5.50899165923976474878998033091193e-01 1.82570774440876770761121861141874e-01 2.66530059635146754359880105766933e-01 2.09060455685651375157263487380987e-02
            1.79710032722074752875585090805544e-01 8.15862277549058850789620578325412e-02 7.38703739523019375923240659176372e-01 2.13322476716037226496247569684783e-02
            8.15862277549058850789620578325412e-02 7.38703739523019375923240659176372e-01 1.79710032722074752875585090805544e-01 2.13322476716037226496247569684783e-02
            7.38703739523019375923240659176372e-01 1.79710032722074752875585090805544e-01 8.15862277549058850789620578325412e-02 2.13322476716037226496247569684783e-02
            1.75807941715677179539412122721842e-01 1.80353640520785951650850620353594e-01 6.43838417763536896565312872553477e-01 2.37493771891846804089620803779326e-02
            1.80353640520785951650850620353594e-01 6.43838417763536896565312872553477e-01 1.75807941715677179539412122721842e-01 2.37493771891846804089620803779326e-02
            6.43838417763536896565312872553477e-01 1.75807941715677179539412122721842e-01 1.80353640520785951650850620353594e-01 2.37493771891846804089620803779326e-02
            4.58731499688783517232337771929451e-01 4.74126007719326725275976741613704e-01 6.71424925918898130028367177146720e-02 2.47770240081320279601495570886982e-02
            4.74126007719326725275976741613704e-01 6.71424925918898130028367177146720e-02 4.58731499688783517232337771929451e-01 2.47770240081320279601495570886982e-02
            6.71424925918898130028367177146720e-02 4.58731499688783517232337771929451e-01 4.74126007719326725275976741613704e-01 2.47770240081320279601495570886982e-02
            7.53196920743743475457065983391658e-02 2.93754105362967754544456511212047e-01 6.30926202562657856276473467005417e-01 2.55855702422726240252703888700125e-02
            2.93754105362967754544456511212047e-01 6.30926202562657856276473467005417e-01 7.53196920743743475457065983391658e-02 2.55855702422726240252703888700125e-02
            6.30926202562657856276473467005417e-01 7.53196920743743475457065983391658e-02 2.93754105362967754544456511212047e-01 2.55855702422726240252703888700125e-02
            1.63305177925384792381180432130350e-01 3.84251187765941693008642232598504e-01 4.52443634308673514610177335271146e-01 3.13108588033011586704290607485746e-02
            3.84251187765941693008642232598504e-01 4.52443634308673514610177335271146e-01 1.63305177925384792381180432130350e-01 3.13108588033011586704290607485746e-02
            4.52443634308673514610177335271146e-01 1.63305177925384792381180432130350e-01 3.84251187765941693008642232598504e-01 3.13108588033011586704290607485746e-02
            3.11813568699614629409211374877486e-01 1.39501525996608172208723885887593e-01 5.48684905303777226137640354863834e-01 3.15903625494375200899455080616463e-02
            1.39501525996608172208723885887593e-01 5.48684905303777226137640354863834e-01 3.11813568699614629409211374877486e-01 3.15903625494375200899455080616463e-02
            5.48684905303777226137640354863834e-01 3.11813568699614629409211374877486e-01 1.39501525996608172208723885887593e-01 3.15903625494375200899455080616463e-02
            3.06048259112027931649890888365917e-01 2.67720653545962083974529832630651e-01 4.26231087342009984375579279003432e-01 3.86191604154902459744569398480962e-02
            2.67720653545962083974529832630651e-01 4.26231087342009984375579279003432e-01 3.06048259112027931649890888365917e-01 3.86191604154902459744569398480962e-02
            4.26231087342009984375579279003432e-01 3.06048259112027931649890888365917e-01 2.67720653545962083974529832630651e-01 3.86191604154902459744569398480962e-02
            ];
        
    case  18
        
        % ALG. DEG.:   18
        % PTS CARD.:   64
        % NEG. W.  :    1
        % OUT PTS. :    0
        % M.E.INF. : 4.9e-16
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333425851918718763045e-01 -1.25845221584873240283641848691332e-01
            9.22437177016372997218063289892598e-03 9.54037638902169304877531885722419e-03 9.81235251840814504120658057217952e-01 1.29727110524691319551338075655167e-03
            9.54037638902169304877531885722419e-03 9.81235251840814504120658057217952e-01 9.22437177016372997218063289892598e-03 1.29727110524691319551338075655167e-03
            9.81235251840814504120658057217952e-01 9.22437177016372997218063289892598e-03 9.54037638902169304877531885722419e-03 1.29727110524691319551338075655167e-03
            1.15881748682225369645459878142901e-02 5.99102344454169738474647033399378e-02 9.28501590686360511739394496544264e-01 4.40434451590567769413420151636274e-03
            5.99102344454169738474647033399378e-02 9.28501590686360511739394496544264e-01 1.15881748682225369645459878142901e-02 4.40434451590567769413420151636274e-03
            9.28501590686360511739394496544264e-01 1.15881748682225369645459878142901e-02 5.99102344454169738474647033399378e-02 4.40434451590567769413420151636274e-03
            1.37109449232486801939501219749218e-01 8.13423222383992221284909618361780e-03 8.54756318543673287990714015904814e-01 4.55380277877178666484647706624855e-03
            8.13423222383992221284909618361780e-03 8.54756318543673287990714015904814e-01 1.37109449232486801939501219749218e-01 4.55380277877178666484647706624855e-03
            8.54756318543673287990714015904814e-01 1.37109449232486801939501219749218e-01 8.13423222383992221284909618361780e-03 4.55380277877178666484647706624855e-03
            5.40746581302322046336250593867589e-02 1.59640348658512051804869713578228e-02 9.29961307003916659574827008327702e-01 5.25221168748300770712233997983276e-03
            1.59640348658512051804869713578228e-02 9.29961307003916659574827008327702e-01 5.40746581302322046336250593867589e-02 5.25221168748300770712233997983276e-03
            9.29961307003916659574827008327702e-01 5.40746581302322046336250593867589e-02 1.59640348658512051804869713578228e-02 5.25221168748300770712233997983276e-03
            1.23562085317423936414726881594106e-02 1.56321458214785352636155835170939e-01 8.31322333253472267600159284484107e-01 7.37463787220985107773163491629020e-03
            1.56321458214785352636155835170939e-01 8.31322333253472267600159284484107e-01 1.23562085317423936414726881594106e-02 7.37463787220985107773163491629020e-03
            8.31322333253472267600159284484107e-01 1.23562085317423936414726881594106e-02 1.56321458214785352636155835170939e-01 7.37463787220985107773163491629020e-03
            2.54443010786652157317888622856117e-01 1.32280360805979747829130843683743e-02 7.32328953132749815857494013471296e-01 8.83478687259323623715001616574227e-03
            1.32280360805979747829130843683743e-02 7.32328953132749815857494013471296e-01 2.54443010786652157317888622856117e-01 8.83478687259323623715001616574227e-03
            7.32328953132749815857494013471296e-01 2.54443010786652157317888622856117e-01 1.32280360805979747829130843683743e-02 8.83478687259323623715001616574227e-03
            1.22006545481682538573098284473417e-02 4.38518077301824849012490403765696e-01 5.49281268150006996009437898464967e-01 9.64777642488273169674162232922754e-03
            4.38518077301824849012490403765696e-01 5.49281268150006996009437898464967e-01 1.22006545481682538573098284473417e-02 9.64777642488273169674162232922754e-03
            5.49281268150006996009437898464967e-01 1.22006545481682538573098284473417e-02 4.38518077301824849012490403765696e-01 9.64777642488273169674162232922754e-03
            3.95592717135403115769065607310040e-01 1.29052660579212063324261450247832e-02 5.91502016806675690041572579502827e-01 9.86709978042134457087986021406323e-03
            1.29052660579212063324261450247832e-02 5.91502016806675690041572579502827e-01 3.95592717135403115769065607310040e-01 9.86709978042134457087986021406323e-03
            5.91502016806675690041572579502827e-01 3.95592717135403115769065607310040e-01 1.29052660579212063324261450247832e-02 9.86709978042134457087986021406323e-03
            1.39894903845946654008525200652002e-02 2.87578430196016754383236957437475e-01 6.98432079419388562868675762729254e-01 1.02607131892317442883877376402779e-02
            2.87578430196016754383236957437475e-01 6.98432079419388562868675762729254e-01 1.39894903845946654008525200652002e-02 1.02607131892317442883877376402779e-02
            6.98432079419388562868675762729254e-01 1.39894903845946654008525200652002e-02 2.87578430196016754383236957437475e-01 1.02607131892317442883877376402779e-02
            6.12426222536310363442346726969845e-02 8.21310314077598468962548849958694e-02 8.56626346338609012676101883698720e-01 1.17966959830773238104839606421592e-02
            8.21310314077598468962548849958694e-02 8.56626346338609012676101883698720e-01 6.12426222536310363442346726969845e-02 1.17966959830773238104839606421592e-02
            8.56626346338609012676101883698720e-01 6.12426222536310363442346726969845e-02 8.21310314077598468962548849958694e-02 1.17966959830773238104839606421592e-02
            1.40187294281714058818977264309069e-01 5.09884646182077044818115041380224e-02 8.08824241100078222821423423738452e-01 1.29855075351113964654548027510828e-02
            5.09884646182077044818115041380224e-02 8.08824241100078222821423423738452e-01 1.40187294281714058818977264309069e-01 1.29855075351113964654548027510828e-02
            8.08824241100078222821423423738452e-01 1.40187294281714058818977264309069e-01 5.09884646182077044818115041380224e-02 1.29855075351113964654548027510828e-02
            6.59534606576724913296061458822805e-02 1.82306457551239170511436782362580e-01 7.51740081791088310403381456126226e-01 1.63182058245727769352217961795759e-02
            1.82306457551239170511436782362580e-01 7.51740081791088310403381456126226e-01 6.59534606576724913296061458822805e-02 1.63182058245727769352217961795759e-02
            7.51740081791088310403381456126226e-01 6.59534606576724913296061458822805e-02 1.82306457551239170511436782362580e-01 1.63182058245727769352217961795759e-02
            6.24139442088075535597369025708758e-02 4.11715469349845597513137818168616e-01 5.25870586441346876682700894889422e-01 1.75256165062230781503949117450247e-02
            4.11715469349845597513137818168616e-01 5.25870586441346876682700894889422e-01 6.24139442088075535597369025708758e-02 1.75256165062230781503949117450247e-02
            5.25870586441346876682700894889422e-01 6.24139442088075535597369025708758e-02 4.11715469349845597513137818168616e-01 1.75256165062230781503949117450247e-02
            7.43017152789951784708222248809761e-02 2.93703944612865197605344746989431e-01 6.31994340108139596168257412500679e-01 1.84118314428711564445517723243029e-02
            2.93703944612865197605344746989431e-01 6.31994340108139596168257412500679e-01 7.43017152789951784708222248809761e-02 1.84118314428711564445517723243029e-02
            6.31994340108139596168257412500679e-01 7.43017152789951784708222248809761e-02 2.93703944612865197605344746989431e-01 1.84118314428711564445517723243029e-02
            2.55758429076780702260407451831270e-01 6.81831468635734766481348856359546e-02 6.76058424059645779458094239089405e-01 1.90689705940650690296767066911343e-02
            6.81831468635734766481348856359546e-02 6.76058424059645779458094239089405e-01 2.55758429076780702260407451831270e-01 1.90689705940650690296767066911343e-02
            6.76058424059645779458094239089405e-01 2.55758429076780702260407451831270e-01 6.81831468635734766481348856359546e-02 1.90689705940650690296767066911343e-02
            3.89748898002750199243138240490225e-01 6.69978935694291871660865922422090e-02 5.43253208427820655224138590710936e-01 2.05821846859529436479796515868657e-02
            6.69978935694291871660865922422090e-02 5.43253208427820655224138590710936e-01 3.89748898002750199243138240490225e-01 2.05821846859529436479796515868657e-02
            5.43253208427820655224138590710936e-01 3.89748898002750199243138240490225e-01 6.69978935694291871660865922422090e-02 2.05821846859529436479796515868657e-02
            1.55168432990029553364408343441028e-01 1.40086682350669539109944139454456e-01 7.04744884659300852014496285846690e-01 2.35891412493054869470299195199914e-02
            1.40086682350669539109944139454456e-01 7.04744884659300852014496285846690e-01 1.55168432990029553364408343441028e-01 2.35891412493054869470299195199914e-02
            7.04744884659300852014496285846690e-01 1.55168432990029553364408343441028e-01 1.40086682350669539109944139454456e-01 2.35891412493054869470299195199914e-02
            1.70472927860750189665850484743714e-01 2.61254002060855416988971455793944e-01 5.68273070078394448856329290720169e-01 3.10909522631880735898945999906573e-02
            2.61254002060855416988971455793944e-01 5.68273070078394448856329290720169e-01 1.70472927860750189665850484743714e-01 3.10909522631880735898945999906573e-02
            5.68273070078394448856329290720169e-01 1.70472927860750189665850484743714e-01 2.61254002060855416988971455793944e-01 3.10909522631880735898945999906573e-02
            2.86469196549182070210548545219353e-01 1.63053894264229143340472205636615e-01 5.50476909186588758693403633515118e-01 3.20399669872254433400371453899425e-02
            1.63053894264229143340472205636615e-01 5.50476909186588758693403633515118e-01 2.86469196549182070210548545219353e-01 3.20399669872254433400371453899425e-02
            5.50476909186588758693403633515118e-01 2.86469196549182070210548545219353e-01 1.63053894264229143340472205636615e-01 3.20399669872254433400371453899425e-02
            1.55079032148931039714057078526821e-01 4.08496920155350773384839158097748e-01 4.36424047695718186901103763375431e-01 3.26604177729548361686084945176844e-02
            4.08496920155350773384839158097748e-01 4.36424047695718186901103763375431e-01 1.55079032148931039714057078526821e-01 3.26604177729548361686084945176844e-02
            4.36424047695718186901103763375431e-01 1.55079032148931039714057078526821e-01 4.08496920155350773384839158097748e-01 3.26604177729548361686084945176844e-02
            3.06409473651319375075985362855135e-01 3.04333597697522717506757317096344e-01 3.89256928651157851906106088790693e-01 7.77196054569972011316636439914873e-02
            3.04333597697522717506757317096344e-01 3.89256928651157851906106088790693e-01 3.06409473651319375075985362855135e-01 7.77196054569972011316636439914873e-02
            3.89256928651157851906106088790693e-01 3.06409473651319375075985362855135e-01 3.04333597697522717506757317096344e-01 7.77196054569972011316636439914873e-02
            ];
        
    case  19
        
        % ALG. DEG.:   19
        % PTS CARD.:   70
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.2e-16
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333425851918718763045e-01 3.04629617601474174393150207151848e-02
            1.03766565457017543444573348665472e-02 1.44810472658998996842161233189472e-02 9.75142296188398316481027450208785e-01 1.95018195806569887354453562977596e-03
            1.44810472658998996842161233189472e-02 9.75142296188398316481027450208785e-01 1.03766565457017543444573348665472e-02 1.95018195806569887354453562977596e-03
            9.75142296188398316481027450208785e-01 1.03766565457017543444573348665472e-02 1.44810472658998996842161233189472e-02 1.95018195806569887354453562977596e-03
            5.28956402129591926608576102353254e-02 8.92711734256453406932685368246894e-03 9.38177242444476200411429545056308e-01 2.76194461717317509691360655210701e-03
            8.92711734256453406932685368246894e-03 9.38177242444476200411429545056308e-01 5.28956402129591926608576102353254e-02 2.76194461717317509691360655210701e-03
            9.38177242444476200411429545056308e-01 5.28956402129591926608576102353254e-02 8.92711734256453406932685368246894e-03 2.76194461717317509691360655210701e-03
            1.30940571465279348428722983044281e-01 6.39606356170804597055523643689412e-03 8.62663364973012547487485335295787e-01 3.53317943173294237227399783307646e-03
            6.39606356170804597055523643689412e-03 8.62663364973012547487485335295787e-01 1.30940571465279348428722983044281e-01 3.53317943173294237227399783307646e-03
            8.62663364973012547487485335295787e-01 1.30940571465279348428722983044281e-01 6.39606356170804597055523643689412e-03 3.53317943173294237227399783307646e-03
            1.26681865510903015475951960411294e-02 7.29082350599041251193455082102446e-02 9.14423578389005564659441915864591e-01 5.15202907769013885436182320631815e-03
            7.29082350599041251193455082102446e-02 9.14423578389005564659441915864591e-01 1.26681865510903015475951960411294e-02 5.15202907769013885436182320631815e-03
            9.14423578389005564659441915864591e-01 1.26681865510903015475951960411294e-02 7.29082350599041251193455082102446e-02 5.15202907769013885436182320631815e-03
            2.46613960104767354630794784497994e-01 1.13210284482716614512520791890893e-02 7.42065011446961020347146131825866e-01 7.31171016627054411868291694531763e-03
            1.13210284482716614512520791890893e-02 7.42065011446961020347146131825866e-01 2.46613960104767354630794784497994e-01 7.31171016627054411868291694531763e-03
            7.42065011446961020347146131825866e-01 2.46613960104767354630794784497994e-01 1.13210284482716614512520791890893e-02 7.31171016627054411868291694531763e-03
            1.24330000962839457012698574089882e-02 1.71507203857275219993638870619179e-01 8.16059796046440855121772983693518e-01 7.38947305176194165626801790835998e-03
            1.71507203857275219993638870619179e-01 8.16059796046440855121772983693518e-01 1.24330000962839457012698574089882e-02 7.38947305176194165626801790835998e-03
            8.16059796046440855121772983693518e-01 1.24330000962839457012698574089882e-02 1.71507203857275219993638870619179e-01 7.38947305176194165626801790835998e-03
            9.45870867926038548512135406554080e-03 4.54553083656948797575125809089513e-01 5.35988207663790849899498880404280e-01 7.54778605267613766222067894773318e-03
            4.54553083656948797575125809089513e-01 5.35988207663790849899498880404280e-01 9.45870867926038548512135406554080e-03 7.54778605267613766222067894773318e-03
            5.35988207663790849899498880404280e-01 9.45870867926038548512135406554080e-03 4.54553083656948797575125809089513e-01 7.54778605267613766222067894773318e-03
            3.82677127432900721082376094273059e-01 1.03041271596045185682255507231275e-02 6.07018745407494786370250494655920e-01 7.61546840667602122076917936510654e-03
            1.03041271596045185682255507231275e-02 6.07018745407494786370250494655920e-01 3.82677127432900721082376094273059e-01 7.61546840667602122076917936510654e-03
            6.07018745407494786370250494655920e-01 3.82677127432900721082376094273059e-01 1.03041271596045185682255507231275e-02 7.61546840667602122076917936510654e-03
            1.07221902311333636831358617769183e-02 3.02819418977863463737776328343898e-01 6.86458390791003125741553958505392e-01 7.90075813312910400776356567575931e-03
            3.02819418977863463737776328343898e-01 6.86458390791003125741553958505392e-01 1.07221902311333636831358617769183e-02 7.90075813312910400776356567575931e-03
            6.86458390791003125741553958505392e-01 1.07221902311333636831358617769183e-02 3.02819418977863463737776328343898e-01 7.90075813312910400776356567575931e-03
            6.34588872173355300621011565453955e-02 4.79826825588268238154121547722752e-02 8.88558430223837625305804976960644e-01 8.21043953292401734733729767867771e-03
            4.79826825588268238154121547722752e-02 8.88558430223837625305804976960644e-01 6.34588872173355300621011565453955e-02 8.21043953292401734733729767867771e-03
            8.88558430223837625305804976960644e-01 6.34588872173355300621011565453955e-02 4.79826825588268238154121547722752e-02 8.21043953292401734733729767867771e-03
            1.49585341667801352549815874226624e-01 4.15353642843105178017459877537476e-02 8.08879294047888164342907657555770e-01 1.06316665808041440483000528161028e-02
            4.15353642843105178017459877537476e-02 8.08879294047888164342907657555770e-01 1.49585341667801352549815874226624e-01 1.06316665808041440483000528161028e-02
            8.08879294047888164342907657555770e-01 1.49585341667801352549815874226624e-01 4.15353642843105178017459877537476e-02 1.06316665808041440483000528161028e-02
            6.59383749931683749156974272409570e-02 1.27098301306499766871738188456220e-01 8.06963323700331858212564384302823e-01 1.45180631220521022539671207596257e-02
            1.27098301306499766871738188456220e-01 8.06963323700331858212564384302823e-01 6.59383749931683749156974272409570e-02 1.45180631220521022539671207596257e-02
            8.06963323700331858212564384302823e-01 6.59383749931683749156974272409570e-02 1.27098301306499766871738188456220e-01 1.45180631220521022539671207596257e-02
            4.19360344474693613747717790829483e-01 4.84133944843579896999408163082990e-02 5.32226261040948389613447488954989e-01 1.46941137319119476817297353932190e-02
            4.84133944843579896999408163082990e-02 5.32226261040948389613447488954989e-01 4.19360344474693613747717790829483e-01 1.46941137319119476817297353932190e-02
            5.32226261040948389613447488954989e-01 4.19360344474693613747717790829483e-01 4.84133944843579896999408163082990e-02 1.46941137319119476817297353932190e-02
            5.12859715627437540153010786525556e-02 3.85451519447595591927324676362332e-01 5.63262508989660681812949860614026e-01 1.68699111933565473742113738353510e-02
            3.85451519447595591927324676362332e-01 5.63262508989660681812949860614026e-01 5.12859715627437540153010786525556e-02 1.68699111933565473742113738353510e-02
            5.63262508989660681812949860614026e-01 5.12859715627437540153010786525556e-02 3.85451519447595591927324676362332e-01 1.68699111933565473742113738353510e-02
            5.92161717125940026407171501432458e-02 2.44263378740914083309121451748069e-01 6.96520449546491948744630917644827e-01 1.72191695246377389494352172505387e-02
            2.44263378740914083309121451748069e-01 6.96520449546491948744630917644827e-01 5.92161717125940026407171501432458e-02 1.72191695246377389494352172505387e-02
            6.96520449546491948744630917644827e-01 5.92161717125940026407171501432458e-02 2.44263378740914083309121451748069e-01 1.72191695246377389494352172505387e-02
            1.57744353652534891052638954533904e-01 1.07039102999093777013328576686035e-01 7.35216543348371276422881237522233e-01 1.74916741454707938552282087130152e-02
            1.07039102999093777013328576686035e-01 7.35216543348371276422881237522233e-01 1.57744353652534891052638954533904e-01 1.74916741454707938552282087130152e-02
            7.35216543348371276422881237522233e-01 1.57744353652534891052638954533904e-01 1.07039102999093777013328576686035e-01 1.74916741454707938552282087130152e-02
            2.76356371451293159058337778333225e-01 5.93958296507460237201314612320857e-02 6.64247798897960928243833222950343e-01 1.82914778002982274240917348606672e-02
            5.93958296507460237201314612320857e-02 6.64247798897960928243833222950343e-01 2.76356371451293159058337778333225e-01 1.82914778002982274240917348606672e-02
            6.64247798897960928243833222950343e-01 2.76356371451293159058337778333225e-01 5.93958296507460237201314612320857e-02 1.82914778002982274240917348606672e-02
            4.02925432504102898345621497355751e-01 1.11259697463848866716418228861585e-01 4.85814870032048151671233426895924e-01 2.10893555024571459355353653108978e-02
            1.11259697463848866716418228861585e-01 4.85814870032048151671233426895924e-01 4.02925432504102898345621497355751e-01 2.10893555024571459355353653108978e-02
            4.85814870032048151671233426895924e-01 4.02925432504102898345621497355751e-01 1.11259697463848866716418228861585e-01 2.10893555024571459355353653108978e-02
            1.44741209510670071347604448419588e-01 2.07337086661877140203458225187205e-01 6.47921703827452843960088557651034e-01 2.30291713310627581268175134709963e-02
            2.07337086661877140203458225187205e-01 6.47921703827452843960088557651034e-01 1.44741209510670071347604448419588e-01 2.30291713310627581268175134709963e-02
            6.47921703827452843960088557651034e-01 1.44741209510670071347604448419588e-01 2.07337086661877140203458225187205e-01 2.30291713310627581268175134709963e-02
            1.28002196170715176748444719123654e-01 3.38803971814336801759992567895097e-01 5.33193832014948077002713944239076e-01 2.49341884513248727983647512473908e-02
            3.38803971814336801759992567895097e-01 5.33193832014948077002713944239076e-01 1.28002196170715176748444719123654e-01 2.49341884513248727983647512473908e-02
            5.33193832014948077002713944239076e-01 1.28002196170715176748444719123654e-01 3.38803971814336801759992567895097e-01 2.49341884513248727983647512473908e-02
            2.67334020957130047424499252883834e-01 1.47872285803578001894820204142889e-01 5.84793693239291978436256158602191e-01 2.59561349491033883929436143489511e-02
            1.47872285803578001894820204142889e-01 5.84793693239291978436256158602191e-01 2.67334020957130047424499252883834e-01 2.59561349491033883929436143489511e-02
            5.84793693239291978436256158602191e-01 2.67334020957130047424499252883834e-01 1.47872285803578001894820204142889e-01 2.59561349491033883929436143489511e-02
            3.80640391337433547924717913701897e-01 2.07421341282564897223750222110539e-01 4.11938267380001554851531864187564e-01 2.89155163062293323050777615890183e-02
            2.07421341282564897223750222110539e-01 4.11938267380001554851531864187564e-01 3.80640391337433547924717913701897e-01 2.89155163062293323050777615890183e-02
            4.11938267380001554851531864187564e-01 3.80640391337433547924717913701897e-01 2.07421341282564897223750222110539e-01 2.89155163062293323050777615890183e-02
            2.41019074102765712419582655456907e-01 2.67083117991290863724174187154858e-01 4.91897807905943451611818773017148e-01 3.01655996798087348842454247233036e-02
            2.67083117991290863724174187154858e-01 4.91897807905943451611818773017148e-01 2.41019074102765712419582655456907e-01 3.01655996798087348842454247233036e-02
            4.91897807905943451611818773017148e-01 2.41019074102765712419582655456907e-01 2.67083117991290863724174187154858e-01 3.01655996798087348842454247233036e-02
            ];
        
    case  20
        
        % ALG. DEG.:   20
        % PTS CARD.:   78
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.9e-16
        
        xyw_bar=[
            7.90368134162346309967706048382752e-03 1.08926176663298908409149845510910e-02 9.81203700992046679019153998524416e-01 1.13899061489750388909136802340072e-03
            1.08926176663298908409149845510910e-02 9.81203700992046679019153998524416e-01 7.90368134162346309967706048382752e-03 1.13899061489750388909136802340072e-03
            9.81203700992046679019153998524416e-01 7.90368134162346309967706048382752e-03 1.08926176663298908409149845510910e-02 1.13899061489750388909136802340072e-03
            3.70261421404957180927852711249670e-03 5.68359139419127579184731757777627e-02 9.39461471844037609990607506915694e-01 1.53576708745824935808954592175724e-03
            5.68359139419127579184731757777627e-02 9.39461471844037609990607506915694e-01 3.70261421404957180927852711249670e-03 1.53576708745824935808954592175724e-03
            9.39461471844037609990607506915694e-01 3.70261421404957180927852711249670e-03 5.68359139419127579184731757777627e-02 1.53576708745824935808954592175724e-03
            1.09337390150228475071969569398789e-01 3.03859257334367343500458602534309e-03 8.87624017276427812461747635097709e-01 1.92468559178653620539423130253454e-03
            3.03859257334367343500458602534309e-03 8.87624017276427812461747635097709e-01 1.09337390150228475071969569398789e-01 1.92468559178653620539423130253454e-03
            8.87624017276427812461747635097709e-01 1.09337390150228475071969569398789e-01 3.03859257334367343500458602534309e-03 1.92468559178653620539423130253454e-03
            4.28810753769432936866046190971247e-02 9.80505702549880342466792626510141e-03 9.47313867597557868194257935101632e-01 2.50043731235591402267681715443359e-03
            9.80505702549880342466792626510141e-03 9.47313867597557868194257935101632e-01 4.28810753769432936866046190971247e-02 2.50043731235591402267681715443359e-03
            9.47313867597557868194257935101632e-01 4.28810753769432936866046190971247e-02 9.80505702549880342466792626510141e-03 2.50043731235591402267681715443359e-03
            1.77473257307326614390374786012217e-03 2.36693018982774450176265190748381e-01 7.61532248444152237709658948006108e-01 2.63075123618766162567061961397030e-03
            2.36693018982774450176265190748381e-01 7.61532248444152237709658948006108e-01 1.77473257307326614390374786012217e-03 2.63075123618766162567061961397030e-03
            7.61532248444152237709658948006108e-01 1.77473257307326614390374786012217e-03 2.36693018982774450176265190748381e-01 2.63075123618766162567061961397030e-03
            3.11814570601137097671795572750852e-01 2.54551979563882565041343930545281e-03 6.85639909603224184664327367499936e-01 3.19562561402815222336148970327940e-03
            2.54551979563882565041343930545281e-03 6.85639909603224184664327367499936e-01 3.11814570601137097671795572750852e-01 3.19562561402815222336148970327940e-03
            6.85639909603224184664327367499936e-01 3.11814570601137097671795572750852e-01 2.54551979563882565041343930545281e-03 3.19562561402815222336148970327940e-03
            2.99000660469557651144345555849213e-03 4.63119441155936895082589899175218e-01 5.33890552239367544018477929057553e-01 3.65176899407884582998695677247269e-03
            4.63119441155936895082589899175218e-01 5.33890552239367544018477929057553e-01 2.99000660469557651144345555849213e-03 3.65176899407884582998695677247269e-03
            5.33890552239367544018477929057553e-01 2.99000660469557651144345555849213e-03 4.63119441155936895082589899175218e-01 3.65176899407884582998695677247269e-03
            3.22142321349414423314883038074186e-02 5.24854182546855979607514086637821e-02 9.15300349610372987463335903157713e-01 5.32453789121690480856674554388519e-03
            5.24854182546855979607514086637821e-02 9.15300349610372987463335903157713e-01 3.22142321349414423314883038074186e-02 5.32453789121690480856674554388519e-03
            9.15300349610372987463335903157713e-01 3.22142321349414423314883038074186e-02 5.24854182546855979607514086637821e-02 5.32453789121690480856674554388519e-03
            1.27799603324881029531301379620345e-02 1.31379538178000160320024747306888e-01 8.55840501489511740196292066684691e-01 5.84445602838894011366033254262220e-03
            1.31379538178000160320024747306888e-01 8.55840501489511740196292066684691e-01 1.27799603324881029531301379620345e-02 5.84445602838894011366033254262220e-03
            8.55840501489511740196292066684691e-01 1.27799603324881029531301379620345e-02 1.31379538178000160320024747306888e-01 5.84445602838894011366033254262220e-03
            1.95430723381915305569833662957535e-01 1.40953400791673909980827872345799e-02 7.90473936538917398841874728532275e-01 7.30800598014199644103516817494892e-03
            1.40953400791673909980827872345799e-02 7.90473936538917398841874728532275e-01 1.95430723381915305569833662957535e-01 7.30800598014199644103516817494892e-03
            7.90473936538917398841874728532275e-01 1.95430723381915305569833662957535e-01 1.40953400791673909980827872345799e-02 7.30800598014199644103516817494892e-03
            9.77669169905016482990944837183633e-02 3.50007516628948259218567784500920e-02 8.67232331346603491084579218295403e-01 7.90633535329192740914727721701638e-03
            3.50007516628948259218567784500920e-02 8.67232331346603491084579218295403e-01 9.77669169905016482990944837183633e-02 7.90633535329192740914727721701638e-03
            8.67232331346603491084579218295403e-01 9.77669169905016482990944837183633e-02 3.50007516628948259218567784500920e-02 7.90633535329192740914727721701638e-03
            1.66006628441778165961029145591965e-02 3.38676752265837932576175717258593e-01 6.44722584889984240419380512321368e-01 9.65595787088912443918431449674245e-03
            3.38676752265837932576175717258593e-01 6.44722584889984240419380512321368e-01 1.66006628441778165961029145591965e-02 9.65595787088912443918431449674245e-03
            6.44722584889984240419380512321368e-01 1.66006628441778165961029145591965e-02 3.38676752265837932576175717258593e-01 9.65595787088912443918431449674245e-03
            4.13395082180740003341412602821947e-01 1.82025878269482105364929225288506e-02 5.68402329992311883266609129350400e-01 1.03145890038596216992106135990070e-02
            1.82025878269482105364929225288506e-02 5.68402329992311883266609129350400e-01 4.13395082180740003341412602821947e-01 1.03145890038596216992106135990070e-02
            5.68402329992311883266609129350400e-01 4.13395082180740003341412602821947e-01 1.82025878269482105364929225288506e-02 1.03145890038596216992106135990070e-02
            6.83293746341590430004586664836097e-02 1.08675089954566678196457019112131e-01 8.22995535411274348192023353476543e-01 1.22098385442625488594980964762726e-02
            1.08675089954566678196457019112131e-01 8.22995535411274348192023353476543e-01 6.83293746341590430004586664836097e-02 1.22098385442625488594980964762726e-02
            8.22995535411274348192023353476543e-01 6.83293746341590430004586664836097e-02 1.08675089954566678196457019112131e-01 1.22098385442625488594980964762726e-02
            3.92153078432324539437381361040025e-02 2.14338430054550499459509182997863e-01 7.46446262102216984146707545733079e-01 1.28386823220992402072582194705319e-02
            2.14338430054550499459509182997863e-01 7.46446262102216984146707545733079e-01 3.92153078432324539437381361040025e-02 1.28386823220992402072582194705319e-02
            7.46446262102216984146707545733079e-01 3.92153078432324539437381361040025e-02 2.14338430054550499459509182997863e-01 1.28386823220992402072582194705319e-02
            2.82851404387685689822973245100002e-01 4.32505997246091511421184350183466e-02 6.73897995887705159034908319881652e-01 1.46667976977237508179330305324584e-02
            4.32505997246091511421184350183466e-02 6.73897995887705159034908319881652e-01 2.82851404387685689822973245100002e-01 1.46667976977237508179330305324584e-02
            6.73897995887705159034908319881652e-01 2.82851404387685689822973245100002e-01 4.32505997246091511421184350183466e-02 1.46667976977237508179330305324584e-02
            1.63149078355705329235192380110675e-01 7.46324203435316663179932561433816e-02 7.62218501300762962813450940302573e-01 1.54050538623656755460311629235548e-02
            7.46324203435316663179932561433816e-02 7.62218501300762962813450940302573e-01 1.63149078355705329235192380110675e-01 1.54050538623656755460311629235548e-02
            7.62218501300762962813450940302573e-01 1.63149078355705329235192380110675e-01 7.46324203435316663179932561433816e-02 1.54050538623656755460311629235548e-02
            4.36290583906621953769189303784515e-01 5.16537077780689246608858411491383e-01 4.71723383126887441108010534662753e-02 1.65755938748782943870097028593591e-02
            5.16537077780689246608858411491383e-01 4.71723383126887441108010534662753e-02 4.36290583906621953769189303784515e-01 1.65755938748782943870097028593591e-02
            4.71723383126887441108010534662753e-02 4.36290583906621953769189303784515e-01 5.16537077780689246608858411491383e-01 1.65755938748782943870097028593591e-02
            1.23069087330845969296966302408691e-01 1.73946464302243086752497447378119e-01 7.02984448366910985583899673656560e-01 1.99892203811951919834033475353863e-02
            1.73946464302243086752497447378119e-01 7.02984448366910985583899673656560e-01 1.23069087330845969296966302408691e-01 1.99892203811951919834033475353863e-02
            7.02984448366910985583899673656560e-01 1.23069087330845969296966302408691e-01 1.73946464302243086752497447378119e-01 1.99892203811951919834033475353863e-02
            8.32579050810742932187125120435667e-02 2.99185227797303610053347711073002e-01 6.17556867121622055094576353440061e-01 2.02974174775074275856034944354178e-02
            2.99185227797303610053347711073002e-01 6.17556867121622055094576353440061e-01 8.32579050810742932187125120435667e-02 2.02974174775074275856034944354178e-02
            6.17556867121622055094576353440061e-01 8.32579050810742932187125120435667e-02 2.99185227797303610053347711073002e-01 2.02974174775074275856034944354178e-02
            3.76028943583988584098420915324823e-01 8.64379444693639392705009072415123e-02 5.37533111946647546020017216505948e-01 2.16366905057804904954910085734809e-02
            8.64379444693639392705009072415123e-02 5.37533111946647546020017216505948e-01 3.76028943583988584098420915324823e-01 2.16366905057804904954910085734809e-02
            5.37533111946647546020017216505948e-01 3.76028943583988584098420915324823e-01 8.64379444693639392705009072415123e-02 2.16366905057804904954910085734809e-02
            2.39866290944955318220621620639577e-01 1.30115610897138883617074611720454e-01 6.30018098157905770406728152011055e-01 2.30254773629517753286144454705209e-02
            1.30115610897138883617074611720454e-01 6.30018098157905770406728152011055e-01 2.39866290944955318220621620639577e-01 2.30254773629517753286144454705209e-02
            6.30018098157905770406728152011055e-01 2.39866290944955318220621620639577e-01 1.30115610897138883617074611720454e-01 2.30254773629517753286144454705209e-02
            1.37221148823993804866105961082212e-01 3.91013201302957691396500194969121e-01 4.71765649873048475981818228319753e-01 2.63234937477725451648868926213254e-02
            3.91013201302957691396500194969121e-01 4.71765649873048475981818228319753e-01 1.37221148823993804866105961082212e-01 2.63234937477725451648868926213254e-02
            4.71765649873048475981818228319753e-01 1.37221148823993804866105961082212e-01 3.91013201302957691396500194969121e-01 2.63234937477725451648868926213254e-02
            1.89932130608380905778531655414554e-01 2.50321346008893164114539331421838e-01 5.59746523382725902351353397534695e-01 2.70480976270349934842407435553469e-02
            2.50321346008893164114539331421838e-01 5.59746523382725902351353397534695e-01 1.89932130608380905778531655414554e-01 2.70480976270349934842407435553469e-02
            5.59746523382725902351353397534695e-01 1.89932130608380905778531655414554e-01 2.50321346008893164114539331421838e-01 2.70480976270349934842407435553469e-02
            3.24158256395638832980665711147594e-01 1.94724488881111906790621901564009e-01 4.81117254723249176961985540401656e-01 2.89943397296065115409380297251118e-02
            1.94724488881111906790621901564009e-01 4.81117254723249176961985540401656e-01 3.24158256395638832980665711147594e-01 2.89943397296065115409380297251118e-02
            4.81117254723249176961985540401656e-01 3.24158256395638832980665711147594e-01 1.94724488881111906790621901564009e-01 2.89943397296065115409380297251118e-02
            3.30862912147051713152023921793443e-01 4.07445324376853323933289630076615e-01 2.61691763476095018425837679387769e-01 3.13907216215734349851196327563230e-02
            4.07445324376853323933289630076615e-01 2.61691763476095018425837679387769e-01 3.30862912147051713152023921793443e-01 3.13907216215734349851196327563230e-02
            2.61691763476095018425837679387769e-01 3.30862912147051713152023921793443e-01 4.07445324376853323933289630076615e-01 3.13907216215734349851196327563230e-02
            ];
        
    case  21
        
        % ALG. DEG.:   21
        % PTS CARD.:   85
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.6e-16
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333425851918718763045e-01 1.59272439217052891313741724843567e-02
            1.08833811258291848766965870254353e-02 1.08437360200908869400926448633982e-02 9.78272882854079872672059536853340e-01 1.54002481041379908274779886312444e-03
            1.08437360200908869400926448633982e-02 9.78272882854079872672059536853340e-01 1.08833811258291848766965870254353e-02 1.54002481041379908274779886312444e-03
            9.78272882854079872672059536853340e-01 1.08833811258291848766965870254353e-02 1.08437360200908869400926448633982e-02 1.54002481041379908274779886312444e-03
            1.34483082013301280532857617799891e-01 5.89401791446114188544047607365428e-03 8.59622900072237627888682709453860e-01 2.96403821123544276849859357980677e-03
            5.89401791446114188544047607365428e-03 8.59622900072237627888682709453860e-01 1.34483082013301280532857617799891e-01 2.96403821123544276849859357980677e-03
            8.59622900072237627888682709453860e-01 1.34483082013301280532857617799891e-01 5.89401791446114188544047607365428e-03 2.96403821123544276849859357980677e-03
            5.60697914456082802114877949861693e-02 9.34072029515549240308303069468820e-01 9.85817903884245172463351991609670e-03 3.14257047937977103402307221813317e-03
            9.34072029515549240308303069468820e-01 9.85817903884245172463351991609670e-03 5.60697914456082802114877949861693e-02 3.14257047937977103402307221813317e-03
            9.85817903884245172463351991609670e-03 5.60697914456082802114877949861693e-02 9.34072029515549240308303069468820e-01 3.14257047937977103402307221813317e-03
            5.62307971638516188028766862316843e-02 1.04111689007461219458328116616030e-02 9.33358033935402287006866117735626e-01 3.32255772951118265312597799265859e-03
            1.04111689007461219458328116616030e-02 9.33358033935402287006866117735626e-01 5.62307971638516188028766862316843e-02 3.32255772951118265312597799265859e-03
            9.33358033935402287006866117735626e-01 5.62307971638516188028766862316843e-02 1.04111689007461219458328116616030e-02 3.32255772951118265312597799265859e-03
            4.95492250545260581873208138858899e-01 5.30139353929082006444684083135144e-03 4.99206355915448596327621544332942e-01 4.07094212767519694068374391804355e-03
            5.30139353929082006444684083135144e-03 4.99206355915448596327621544332942e-01 4.95492250545260581873208138858899e-01 4.07094212767519694068374391804355e-03
            4.99206355915448596327621544332942e-01 4.95492250545260581873208138858899e-01 5.30139353929082006444684083135144e-03 4.07094212767519694068374391804355e-03
            1.33413544221436503045552512958238e-01 8.56502202113258115367955269903177e-01 1.00842536653054093420678327674977e-02 4.70658038914133561303465569380933e-03
            8.56502202113258115367955269903177e-01 1.00842536653054093420678327674977e-02 1.33413544221436503045552512958238e-01 4.70658038914133561303465569380933e-03
            1.00842536653054093420678327674977e-02 1.33413544221436503045552512958238e-01 8.56502202113258115367955269903177e-01 4.70658038914133561303465569380933e-03
            2.37788853273795369736021143580729e-01 7.53553830034851435115683671028819e-01 8.65731669135316739271956976153888e-03 5.18588023962721512277740387730773e-03
            7.53553830034851435115683671028819e-01 8.65731669135316739271956976153888e-03 2.37788853273795369736021143580729e-01 5.18588023962721512277740387730773e-03
            8.65731669135316739271956976153888e-03 2.37788853273795369736021143580729e-01 7.53553830034851435115683671028819e-01 5.18588023962721512277740387730773e-03
            2.36675822687569759050774109709891e-01 1.04315998683223633974881394692602e-02 7.52892577444107868878120370936813e-01 6.05317227294636852480413935495562e-03
            1.04315998683223633974881394692602e-02 7.52892577444107868878120370936813e-01 2.36675822687569759050774109709891e-01 6.05317227294636852480413935495562e-03
            7.52892577444107868878120370936813e-01 2.36675822687569759050774109709891e-01 1.04315998683223633974881394692602e-02 6.05317227294636852480413935495562e-03
            3.59237431543587948379325780479121e-01 9.77441311908519919193771841037233e-03 6.30988155337326772631456606177380e-01 6.49737315478060399598358998218828e-03
            9.77441311908519919193771841037233e-03 6.30988155337326772631456606177380e-01 3.59237431543587948379325780479121e-01 6.49737315478060399598358998218828e-03
            6.30988155337326772631456606177380e-01 3.59237431543587948379325780479121e-01 9.77441311908519919193771841037233e-03 6.49737315478060399598358998218828e-03
            3.62123795542131587943401882512262e-01 6.27482731394552795123331634385977e-01 1.03934730633156169332664831017610e-02 6.97332416543433673133955963407971e-03
            6.27482731394552795123331634385977e-01 1.03934730633156169332664831017610e-02 3.62123795542131587943401882512262e-01 6.97332416543433673133955963407971e-03
            1.03934730633156169332664831017610e-02 3.62123795542131587943401882512262e-01 6.27482731394552795123331634385977e-01 6.97332416543433673133955963407971e-03
            5.47064625266307918982633395899029e-02 8.93376258603483019982149926363491e-01 5.19172788698861742417989262321498e-02 7.20587558934637339297379909908159e-03
            8.93376258603483019982149926363491e-01 5.19172788698861742417989262321498e-02 5.47064625266307918982633395899029e-02 7.20587558934637339297379909908159e-03
            5.19172788698861742417989262321498e-02 5.47064625266307918982633395899029e-02 8.93376258603483019982149926363491e-01 7.20587558934637339297379909908159e-03
            1.28497845564646773208750119010801e-01 3.68128421478705475378134792663332e-02 8.34689312287482665375648593908409e-01 8.36501282887073226157692573679014e-03
            3.68128421478705475378134792663332e-02 8.34689312287482665375648593908409e-01 1.28497845564646773208750119010801e-01 8.36501282887073226157692573679014e-03
            8.34689312287482665375648593908409e-01 1.28497845564646773208750119010801e-01 3.68128421478705475378134792663332e-02 8.36501282887073226157692573679014e-03
            1.32045621485988418086776619020384e-01 8.16750542291406311790069594280794e-01 5.12038362226052701231537866988219e-02 1.02265963368377808817566432253443e-02
            8.16750542291406311790069594280794e-01 5.12038362226052701231537866988219e-02 1.32045621485988418086776619020384e-01 1.02265963368377808817566432253443e-02
            5.12038362226052701231537866988219e-02 1.32045621485988418086776619020384e-01 8.16750542291406311790069594280794e-01 1.02265963368377808817566432253443e-02
            4.80002887465515781784120008524042e-01 3.46233167655783727334473098835588e-02 4.85373795768905824665750969870714e-01 1.17409287365970831096273485627535e-02
            3.46233167655783727334473098835588e-02 4.85373795768905824665750969870714e-01 4.80002887465515781784120008524042e-01 1.17409287365970831096273485627535e-02
            4.85373795768905824665750969870714e-01 4.80002887465515781784120008524042e-01 3.46233167655783727334473098835588e-02 1.17409287365970831096273485627535e-02
            2.35698652486183879428693899171776e-01 7.17639780294608065069894564658171e-01 4.66615672192079999902603049122263e-02 1.21832494441335686258254256131295e-02
            7.17639780294608065069894564658171e-01 4.66615672192079999902603049122263e-02 2.35698652486183879428693899171776e-01 1.21832494441335686258254256131295e-02
            4.66615672192079999902603049122263e-02 2.35698652486183879428693899171776e-01 7.17639780294608065069894564658171e-01 1.21832494441335686258254256131295e-02
            1.21235453505120910344672324754356e-01 9.87172892132079649041287439104053e-02 7.80047257281671124751198931335239e-01 1.28654894748754123484646427755251e-02
            9.87172892132079649041287439104053e-02 7.80047257281671124751198931335239e-01 1.21235453505120910344672324754356e-01 1.28654894748754123484646427755251e-02
            7.80047257281671124751198931335239e-01 1.21235453505120910344672324754356e-01 9.87172892132079649041287439104053e-02 1.28654894748754123484646427755251e-02
            2.26921312534569846741661081068742e-01 5.33826956732506574532237664243439e-02 7.19695991792179468049539536878001e-01 1.30952982620168266458815509167835e-02
            5.33826956732506574532237664243439e-02 7.19695991792179468049539536878001e-01 2.26921312534569846741661081068742e-01 1.30952982620168266458815509167835e-02
            7.19695991792179468049539536878001e-01 2.26921312534569846741661081068742e-01 5.33826956732506574532237664243439e-02 1.30952982620168266458815509167835e-02
            3.52725956002229834584937862018705e-01 5.93956020138616369585804477537749e-01 5.33180238591538513404088917013723e-02 1.40867190533422448339706534170546e-02
            5.93956020138616369585804477537749e-01 5.33180238591538513404088917013723e-02 3.52725956002229834584937862018705e-01 1.40867190533422448339706534170546e-02
            5.33180238591538513404088917013723e-02 3.52725956002229834584937862018705e-01 5.93956020138616369585804477537749e-01 1.40867190533422448339706534170546e-02
            3.46394760980635263436511195322964e-01 5.11343671910869101981944595536334e-02 6.02470871828277854120869960752316e-01 1.41840307195744228657696694995138e-02
            5.11343671910869101981944595536334e-02 6.02470871828277854120869960752316e-01 3.46394760980635263436511195322964e-01 1.41840307195744228657696694995138e-02
            6.02470871828277854120869960752316e-01 3.46394760980635263436511195322964e-01 5.11343671910869101981944595536334e-02 1.41840307195744228657696694995138e-02
            2.08545376179994440546394685043197e-01 1.26661874088773246382899628770247e-01 6.64792749731232257559554454928730e-01 1.72208315676023321350029959830863e-02
            1.26661874088773246382899628770247e-01 6.64792749731232257559554454928730e-01 2.08545376179994440546394685043197e-01 1.72208315676023321350029959830863e-02
            6.64792749731232257559554454928730e-01 2.08545376179994440546394685043197e-01 1.26661874088773246382899628770247e-01 1.72208315676023321350029959830863e-02
            1.16641181787498937616476268885890e-01 1.93583513707742599363470503703866e-01 6.89775304504758435264477611781331e-01 1.74136239593561253924391962755180e-02
            1.93583513707742599363470503703866e-01 6.89775304504758435264477611781331e-01 1.16641181787498937616476268885890e-01 1.74136239593561253924391962755180e-02
            6.89775304504758435264477611781331e-01 1.16641181787498937616476268885890e-01 1.93583513707742599363470503703866e-01 1.74136239593561253924391962755180e-02
            3.01681731659511120913208515048609e-01 5.79485561285726924474204224679852e-01 1.18832707054762010123738491529366e-01 1.80836755436484704351318697490569e-02
            5.79485561285726924474204224679852e-01 1.18832707054762010123738491529366e-01 3.01681731659511120913208515048609e-01 1.80836755436484704351318697490569e-02
            1.18832707054762010123738491529366e-01 3.01681731659511120913208515048609e-01 5.79485561285726924474204224679852e-01 1.80836755436484704351318697490569e-02
            3.59766505094364252848038177035050e-01 4.65449578776416461600717866531340e-01 1.74783916129219285551243956433609e-01 1.91491387666732279515535708469542e-02
            4.65449578776416461600717866531340e-01 1.74783916129219285551243956433609e-01 3.59766505094364252848038177035050e-01 1.91491387666732279515535708469542e-02
            1.74783916129219285551243956433609e-01 3.59766505094364252848038177035050e-01 4.65449578776416461600717866531340e-01 1.91491387666732279515535708469542e-02
            3.17727894779720854145921293820720e-01 1.21922765864560744830669136717916e-01 5.60349339355718401023409569461364e-01 1.92278192706125047872767197532085e-02
            1.21922765864560744830669136717916e-01 5.60349339355718401023409569461364e-01 3.17727894779720854145921293820720e-01 1.92278192706125047872767197532085e-02
            5.60349339355718401023409569461364e-01 3.17727894779720854145921293820720e-01 1.21922765864560744830669136717916e-01 1.92278192706125047872767197532085e-02
            4.50599237708166233673523493052926e-01 9.80260802306452266874359224857471e-02 4.51374682061188525761252776646870e-01 2.00316767614215317239168001606231e-02
            9.80260802306452266874359224857471e-02 4.51374682061188525761252776646870e-01 4.50599237708166233673523493052926e-01 2.00316767614215317239168001606231e-02
            4.51374682061188525761252776646870e-01 4.50599237708166233673523493052926e-01 9.80260802306452266874359224857471e-02 2.00316767614215317239168001606231e-02
            2.66780208878214575296539123883122e-01 3.09304671639405781569109876727453e-01 4.23915119482379698645502230647253e-01 2.07651228614931342153493432078903e-02
            3.09304671639405781569109876727453e-01 4.23915119482379698645502230647253e-01 2.66780208878214575296539123883122e-01 2.07651228614931342153493432078903e-02
            4.23915119482379698645502230647253e-01 2.66780208878214575296539123883122e-01 3.09304671639405781569109876727453e-01 2.07651228614931342153493432078903e-02
            2.14913322085512142889385245325684e-01 2.23615271231608347957120486171334e-01 5.61471406682879425886767421616241e-01 2.38150527063891379042637197471777e-02
            2.23615271231608347957120486171334e-01 5.61471406682879425886767421616241e-01 2.14913322085512142889385245325684e-01 2.38150527063891379042637197471777e-02
            5.61471406682879425886767421616241e-01 2.14913322085512142889385245325684e-01 2.23615271231608347957120486171334e-01 2.38150527063891379042637197471777e-02
            3.41820828914985830326855875682668e-01 2.03059575102732176032915845098614e-01 4.55119595982282021395803894847631e-01 2.39076465631621569440934393924181e-02
            2.03059575102732176032915845098614e-01 4.55119595982282021395803894847631e-01 3.41820828914985830326855875682668e-01 2.39076465631621569440934393924181e-02
            4.55119595982282021395803894847631e-01 3.41820828914985830326855875682668e-01 2.03059575102732176032915845098614e-01 2.39076465631621569440934393924181e-02
            ];
        
    case  22
        
        % ALG. DEG.:   22
        % PTS CARD.:   93
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 7.6e-16
        
        xyw_bar=[
            1.17033848378872841304587648636470e-03 4.78536918807590805347018658721936e-02 9.50975969635452167416644897457445e-01 9.75412040939737322829683474623153e-04
            4.78536918807590805347018658721936e-02 9.50975969635452167416644897457445e-01 1.17033848378872841304587648636470e-03 9.75412040939737322829683474623153e-04
            9.50975969635452167416644897457445e-01 1.17033848378872841304587648636470e-03 4.78536918807590805347018658721936e-02 9.75412040939737322829683474623153e-04
            1.09943655740260652631246074406590e-02 9.10867175372020998480593334534205e-03 9.79896962672253679649259083817014e-01 1.31044522114979024654535244565068e-03
            9.10867175372020998480593334534205e-03 9.79896962672253679649259083817014e-01 1.09943655740260652631246074406590e-02 1.31044522114979024654535244565068e-03
            9.79896962672253679649259083817014e-01 1.09943655740260652631246074406590e-02 9.10867175372020998480593334534205e-03 1.31044522114979024654535244565068e-03
            1.06816925225288752131225233910072e-03 2.02709063387925630950192612544924e-01 7.96222767359821559374211119575193e-01 1.81938408464920133890407871746220e-03
            2.02709063387925630950192612544924e-01 7.96222767359821559374211119575193e-01 1.06816925225288752131225233910072e-03 1.81938408464920133890407871746220e-03
            7.96222767359821559374211119575193e-01 1.06816925225288752131225233910072e-03 2.02709063387925630950192612544924e-01 1.81938408464920133890407871746220e-03
            2.21228350592110340616658348267265e-03 4.21769070671465406974220968550071e-01 5.76018645822613528650890657445416e-01 2.48538971789834164244359371309656e-03
            4.21769070671465406974220968550071e-01 5.76018645822613528650890657445416e-01 2.21228350592110340616658348267265e-03 2.48538971789834164244359371309656e-03
            5.76018645822613528650890657445416e-01 2.21228350592110340616658348267265e-03 4.21769070671465406974220968550071e-01 2.48538971789834164244359371309656e-03
            5.72025987162534596541796361179877e-02 8.56087099920958480292476622253162e-03 9.34236530284537014523493780870922e-01 2.79430576242717648022262366680479e-03
            8.56087099920958480292476622253162e-03 9.34236530284537014523493780870922e-01 5.72025987162534596541796361179877e-02 2.79430576242717648022262366680479e-03
            9.34236530284537014523493780870922e-01 5.72025987162534596541796361179877e-02 8.56087099920958480292476622253162e-03 2.79430576242717648022262366680479e-03
            3.53024211615276806153218558392837e-01 6.09424125055589051280779955277467e-03 6.40881547134167228740864175051684e-01 3.54814857201149989884836344344876e-03
            6.09424125055589051280779955277467e-03 6.40881547134167228740864175051684e-01 3.53024211615276806153218558392837e-01 3.54814857201149989884836344344876e-03
            6.40881547134167228740864175051684e-01 3.53024211615276806153218558392837e-01 6.09424125055589051280779955277467e-03 3.54814857201149989884836344344876e-03
            1.36788957839153818563104891836701e-01 7.84998062848008687497625146534119e-03 8.55361061532366129256388376234099e-01 3.77862665631837528826264538395208e-03
            7.84998062848008687497625146534119e-03 8.55361061532366129256388376234099e-01 1.36788957839153818563104891836701e-01 3.77862665631837528826264538395208e-03
            8.55361061532366129256388376234099e-01 1.36788957839153818563104891836701e-01 7.84998062848008687497625146534119e-03 3.77862665631837528826264538395208e-03
            2.41209563760037715152151349684573e-01 7.02726355172032801643666743984795e-03 7.51763172688241887442472943803295e-01 4.03826628503974396910969346663478e-03
            7.02726355172032801643666743984795e-03 7.51763172688241887442472943803295e-01 2.41209563760037715152151349684573e-01 4.03826628503974396910969346663478e-03
            7.51763172688241887442472943803295e-01 2.41209563760037715152151349684573e-01 7.02726355172032801643666743984795e-03 4.03826628503974396910969346663478e-03
            1.16737477866878831822550210972622e-02 1.12641216429535215293178396223084e-01 8.75685035783776988260740381520009e-01 4.63321359302336800017885209967972e-03
            1.12641216429535215293178396223084e-01 8.75685035783776988260740381520009e-01 1.16737477866878831822550210972622e-02 4.63321359302336800017885209967972e-03
            8.75685035783776988260740381520009e-01 1.16737477866878831822550210972622e-02 1.12641216429535215293178396223084e-01 4.63321359302336800017885209967972e-03
            3.01148996347992969901952164946124e-02 4.60558965203824444145652705628891e-02 9.23829203844818258595239512942499e-01 4.93072634030214482919740603961145e-03
            4.60558965203824444145652705628891e-02 9.23829203844818258595239512942499e-01 3.01148996347992969901952164946124e-02 4.93072634030214482919740603961145e-03
            9.23829203844818258595239512942499e-01 3.01148996347992969901952164946124e-02 4.60558965203824444145652705628891e-02 4.93072634030214482919740603961145e-03
            4.56500154387454060156414925586432e-01 1.14379077070311217789733149174936e-02 5.32061937905514814595164807542460e-01 6.41570399077746791188792130355978e-03
            1.14379077070311217789733149174936e-02 5.32061937905514814595164807542460e-01 4.56500154387454060156414925586432e-01 6.41570399077746791188792130355978e-03
            5.32061937905514814595164807542460e-01 4.56500154387454060156414925586432e-01 1.14379077070311217789733149174936e-02 6.41570399077746791188792130355978e-03
            1.18075946189018649024049523177382e-02 3.04996921690560307549588969777687e-01 6.83195483690537841425793885719031e-01 6.64842504414692777453543826027271e-03
            3.04996921690560307549588969777687e-01 6.83195483690537841425793885719031e-01 1.18075946189018649024049523177382e-02 6.64842504414692777453543826027271e-03
            6.83195483690537841425793885719031e-01 1.18075946189018649024049523177382e-02 3.04996921690560307549588969777687e-01 6.64842504414692777453543826027271e-03
            9.61505587301470371341949316956743e-02 4.26352318690261661293661177296599e-02 8.61214209400826824492014566203579e-01 7.93735445568833665463959192720722e-03
            4.26352318690261661293661177296599e-02 8.61214209400826824492014566203579e-01 9.61505587301470371341949316956743e-02 7.93735445568833665463959192720722e-03
            8.61214209400826824492014566203579e-01 9.61505587301470371341949316956743e-02 4.26352318690261661293661177296599e-02 7.93735445568833665463959192720722e-03
            3.06376574992079410419965057599256e-02 1.96990580025709072220507778183674e-01 7.72371762475082990206942668010015e-01 9.39182567120321228071322394725939e-03
            1.96990580025709072220507778183674e-01 7.72371762475082990206942668010015e-01 3.06376574992079410419965057599256e-02 9.39182567120321228071322394725939e-03
            7.72371762475082990206942668010015e-01 3.06376574992079410419965057599256e-02 1.96990580025709072220507778183674e-01 9.39182567120321228071322394725939e-03
            1.94815244430806083775209458508471e-01 3.86641674378723973481974951482698e-02 7.66520588131321511937699142436031e-01 9.76658462379095121397298839838186e-03
            3.86641674378723973481974951482698e-02 7.66520588131321511937699142436031e-01 1.94815244430806083775209458508471e-01 9.76658462379095121397298839838186e-03
            7.66520588131321511937699142436031e-01 1.94815244430806083775209458508471e-01 3.86641674378723973481974951482698e-02 9.76658462379095121397298839838186e-03
            6.03655030702410985710137936166575e-02 1.06610220689147924821860158317577e-01 8.33024276240610928034868720715167e-01 9.93931223641925792877671597125300e-03
            1.06610220689147924821860158317577e-01 8.33024276240610928034868720715167e-01 6.03655030702410985710137936166575e-02 9.93931223641925792877671597125300e-03
            8.33024276240610928034868720715167e-01 6.03655030702410985710137936166575e-02 1.06610220689147924821860158317577e-01 9.93931223641925792877671597125300e-03
            3.15090899482059372260778218333144e-01 3.60898201743391419671347364328540e-02 6.48819280343601478833193141326774e-01 1.07857179503410734178192598164969e-02
            3.60898201743391419671347364328540e-02 6.48819280343601478833193141326774e-01 3.15090899482059372260778218333144e-01 1.07857179503410734178192598164969e-02
            6.48819280343601478833193141326774e-01 3.15090899482059372260778218333144e-01 3.60898201743391419671347364328540e-02 1.07857179503410734178192598164969e-02
            4.11358367215817199991079178289510e-01 5.56390353961855654851831332052825e-01 3.22512788223271451570894896576647e-02 1.11987138171200303893293437340617e-02
            5.56390353961855654851831332052825e-01 3.22512788223271451570894896576647e-02 4.11358367215817199991079178289510e-01 1.11987138171200303893293437340617e-02
            3.22512788223271451570894896576647e-02 4.11358367215817199991079178289510e-01 5.56390353961855654851831332052825e-01 1.11987138171200303893293437340617e-02
            1.44020350630489235577158524392871e-01 9.77155901845127572435956153640291e-02 7.58264059184998062690397091500927e-01 1.35924492506738405633859301246957e-02
            9.77155901845127572435956153640291e-02 7.58264059184998062690397091500927e-01 1.44020350630489235577158524392871e-01 1.35924492506738405633859301246957e-02
            7.58264059184998062690397091500927e-01 1.44020350630489235577158524392871e-01 9.77155901845127572435956153640291e-02 1.35924492506738405633859301246957e-02
            4.20757709102121901789672619997873e-01 5.55288374894526703950958790301229e-02 5.23713453408425344548504654085264e-01 1.41072873131625007442346841912695e-02
            5.55288374894526703950958790301229e-02 5.23713453408425344548504654085264e-01 4.20757709102121901789672619997873e-01 1.41072873131625007442346841912695e-02
            5.23713453408425344548504654085264e-01 4.20757709102121901789672619997873e-01 5.55288374894526703950958790301229e-02 1.41072873131625007442346841912695e-02
            6.09498995807896612464382712914812e-02 2.92056016141072904979125723912148e-01 6.46994084278137426835542100889143e-01 1.43454989859199497775898990425958e-02
            2.92056016141072904979125723912148e-01 6.46994084278137426835542100889143e-01 6.09498995807896612464382712914812e-02 1.43454989859199497775898990425958e-02
            6.46994084278137426835542100889143e-01 6.09498995807896612464382712914812e-02 2.92056016141072904979125723912148e-01 1.43454989859199497775898990425958e-02
            9.69351064455874905645416106381163e-02 1.84224156891908047439798679079104e-01 7.18840736662504475873447518097237e-01 1.50331446811143264186139134608311e-02
            1.84224156891908047439798679079104e-01 7.18840736662504475873447518097237e-01 9.69351064455874905645416106381163e-02 1.50331446811143264186139134608311e-02
            7.18840736662504475873447518097237e-01 9.69351064455874905645416106381163e-02 1.84224156891908047439798679079104e-01 1.50331446811143264186139134608311e-02
            2.53783465620646964122641975336592e-01 9.22286849014402332347373203447205e-02 6.53987849477912774887045088689774e-01 1.58376778038056276187717230641283e-02
            9.22286849014402332347373203447205e-02 6.53987849477912774887045088689774e-01 2.53783465620646964122641975336592e-01 1.58376778038056276187717230641283e-02
            6.53987849477912774887045088689774e-01 2.53783465620646964122641975336592e-01 9.22286849014402332347373203447205e-02 1.58376778038056276187717230641283e-02
            3.48867101402162760326319812520524e-01 1.16235552070709735605902324095950e-01 5.34897346527127615090080325899180e-01 1.71145055482667535207141895625682e-02
            1.16235552070709735605902324095950e-01 5.34897346527127615090080325899180e-01 3.48867101402162760326319812520524e-01 1.71145055482667535207141895625682e-02
            5.34897346527127615090080325899180e-01 3.48867101402162760326319812520524e-01 1.16235552070709735605902324095950e-01 1.71145055482667535207141895625682e-02
            1.90581558653074217835055037539860e-01 1.72148246666744941801852064600098e-01 6.37270194680180868118668513488956e-01 1.79712911006388413592826935882840e-02
            1.72148246666744941801852064600098e-01 6.37270194680180868118668513488956e-01 1.90581558653074217835055037539860e-01 1.79712911006388413592826935882840e-02
            6.37270194680180868118668513488956e-01 1.90581558653074217835055037539860e-01 1.72148246666744941801852064600098e-01 1.79712911006388413592826935882840e-02
            9.69881143348128027525945071829483e-02 3.96552131508626504174941374003538e-01 5.06459754156560748583615350071341e-01 1.85450331969024689859892873755598e-02
            3.96552131508626504174941374003538e-01 5.06459754156560748583615350071341e-01 9.69881143348128027525945071829483e-02 1.85450331969024689859892873755598e-02
            5.06459754156560748583615350071341e-01 9.69881143348128027525945071829483e-02 3.96552131508626504174941374003538e-01 1.85450331969024689859892873755598e-02
            1.41266334630631706925640855843085e-01 2.75379650245183016377126250517904e-01 5.83354015124185165674930431123357e-01 1.93981112463580615545843954805605e-02
            2.75379650245183016377126250517904e-01 5.83354015124185165674930431123357e-01 1.41266334630631706925640855843085e-01 1.93981112463580615545843954805605e-02
            5.83354015124185165674930431123357e-01 1.41266334630631706925640855843085e-01 2.75379650245183016377126250517904e-01 1.93981112463580615545843954805605e-02
            4.12018101979952200419887731186463e-01 1.68573884290088321646905455963861e-01 4.19408013729959394666479965962935e-01 1.95236705864015071265615830498064e-02
            1.68573884290088321646905455963861e-01 4.19408013729959394666479965962935e-01 4.12018101979952200419887731186463e-01 1.95236705864015071265615830498064e-02
            4.19408013729959394666479965962935e-01 4.12018101979952200419887731186463e-01 1.68573884290088321646905455963861e-01 1.95236705864015071265615830498064e-02
            2.76209339065321413109188597445609e-01 1.99199356373184033319745367407450e-01 5.24591304561494498059914803889114e-01 2.06255319696370398308005178478197e-02
            1.99199356373184033319745367407450e-01 5.24591304561494498059914803889114e-01 2.76209339065321413109188597445609e-01 2.06255319696370398308005178478197e-02
            5.24591304561494498059914803889114e-01 2.76209339065321413109188597445609e-01 1.99199356373184033319745367407450e-01 2.06255319696370398308005178478197e-02
            2.18610614672497399757489233707020e-01 3.05685651047730433393923021867522e-01 4.75703734279772139093012128796545e-01 2.13138224201828534176694773805139e-02
            3.05685651047730433393923021867522e-01 4.75703734279772139093012128796545e-01 2.18610614672497399757489233707020e-01 2.13138224201828534176694773805139e-02
            4.75703734279772139093012128796545e-01 2.18610614672497399757489233707020e-01 3.05685651047730433393923021867522e-01 2.13138224201828534176694773805139e-02
            3.36799046311846372869780452674604e-01 2.69814345781991038464298071630765e-01 3.93386607906162533154770244436804e-01 2.35277531670229217430900803265104e-02
            2.69814345781991038464298071630765e-01 3.93386607906162533154770244436804e-01 3.36799046311846372869780452674604e-01 2.35277531670229217430900803265104e-02
            3.93386607906162533154770244436804e-01 3.36799046311846372869780452674604e-01 2.69814345781991038464298071630765e-01 2.35277531670229217430900803265104e-02
            ];
        
    case  23
        
        % ALG. DEG.:   23
        % PTS CARD.:  100
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.7e-16
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333425851918718763045e-01 2.51388714602936956532541756814680e-02
            9.41394903300460904327362499088849e-03 8.64644445189447809207994311009315e-03 9.81939606515100926742434239713475e-01 1.06611768250602771158919157556966e-03
            8.64644445189447809207994311009315e-03 9.81939606515100926742434239713475e-01 9.41394903300460904327362499088849e-03 1.06611768250602771158919157556966e-03
            9.81939606515100926742434239713475e-01 9.41394903300460904327362499088849e-03 8.64644445189447809207994311009315e-03 1.06611768250602771158919157556966e-03
            4.82410218908890808548611817130336e-04 1.08821420580560865309927010002866e-01 8.90696169200530185605657607084140e-01 1.10987315055489140641631351513752e-03
            1.08821420580560865309927010002866e-01 8.90696169200530185605657607084140e-01 4.82410218908890808548611817130336e-04 1.10987315055489140641631351513752e-03
            8.90696169200530185605657607084140e-01 4.82410218908890808548611817130336e-04 1.08821420580560865309927010002866e-01 1.10987315055489140641631351513752e-03
            4.96335565629436092915938161240774e-01 5.03294735930981729232769339432707e-01 3.69698439582122340141268068691716e-04 1.62962395564562878558234615411493e-03
            5.03294735930981729232769339432707e-01 3.69698439582122340141268068691716e-04 4.96335565629436092915938161240774e-01 1.62962395564562878558234615411493e-03
            3.69698439582122340141268068691716e-04 4.96335565629436092915938161240774e-01 5.03294735930981729232769339432707e-01 1.62962395564562878558234615411493e-03
            8.61452759024325040448921697588958e-03 4.49648589409331542188219543731975e-02 9.46420613468823557212772357161157e-01 2.20529169313777488950489846786240e-03
            4.49648589409331542188219543731975e-02 9.46420613468823557212772357161157e-01 8.61452759024325040448921697588958e-03 2.20529169313777488950489846786240e-03
            9.46420613468823557212772357161157e-01 8.61452759024325040448921697588958e-03 4.49648589409331542188219543731975e-02 2.20529169313777488950489846786240e-03
            4.91006634687758056201367651283363e-02 8.24249708242969815474854300418883e-03 9.42656839448794414693111320957541e-01 2.31755431979663913816258613564969e-03
            8.24249708242969815474854300418883e-03 9.42656839448794414693111320957541e-01 4.91006634687758056201367651283363e-02 2.31755431979663913816258613564969e-03
            9.42656839448794414693111320957541e-01 4.91006634687758056201367651283363e-02 8.24249708242969815474854300418883e-03 2.31755431979663913816258613564969e-03
            2.66041607735387425695550156490299e-03 2.94704482021720892692684401481529e-01 7.02635101900925151952037595037837e-01 2.32163870161540206171757461106608e-03
            2.94704482021720892692684401481529e-01 7.02635101900925151952037595037837e-01 2.66041607735387425695550156490299e-03 2.32163870161540206171757461106608e-03
            7.02635101900925151952037595037837e-01 2.66041607735387425695550156490299e-03 2.94704482021720892692684401481529e-01 2.32163870161540206171757461106608e-03
            3.30984469269412107106376197407371e-01 3.82066849179892769131083873901389e-03 6.65194862238788875430373082053848e-01 3.00366151139556600280222653509554e-03
            3.82066849179892769131083873901389e-03 6.65194862238788875430373082053848e-01 3.30984469269412107106376197407371e-01 3.00366151139556600280222653509554e-03
            6.65194862238788875430373082053848e-01 3.30984469269412107106376197407371e-01 3.82066849179892769131083873901389e-03 3.00366151139556600280222653509554e-03
            1.18154505921019392644666368141770e-01 8.41315741466065573905819263700323e-03 8.73432336664319963759339771058876e-01 3.54762192353746518697077405590790e-03
            8.41315741466065573905819263700323e-03 8.73432336664319963759339771058876e-01 1.18154505921019392644666368141770e-01 3.54762192353746518697077405590790e-03
            8.73432336664319963759339771058876e-01 1.18154505921019392644666368141770e-01 8.41315741466065573905819263700323e-03 3.54762192353746518697077405590790e-03
            2.14073097879028895196285020574578e-01 7.20539757650225185653392045992405e-03 7.78721504544468867692330604768358e-01 3.80265163654702493395420859201295e-03
            7.20539757650225185653392045992405e-03 7.78721504544468867692330604768358e-01 2.14073097879028895196285020574578e-01 3.80265163654702493395420859201295e-03
            7.78721504544468867692330604768358e-01 2.14073097879028895196285020574578e-01 7.20539757650225185653392045992405e-03 3.80265163654702493395420859201295e-03
            4.45774128773336539288685287374392e-02 4.30263570230853334419585110026674e-02 9.12396230099581040384748575888807e-01 4.73432470265694373828813112936587e-03
            4.30263570230853334419585110026674e-02 9.12396230099581040384748575888807e-01 4.45774128773336539288685287374392e-02 4.73432470265694373828813112936587e-03
            9.12396230099581040384748575888807e-01 4.45774128773336539288685287374392e-02 4.30263570230853334419585110026674e-02 4.73432470265694373828813112936587e-03
            1.91431395153490890859160344916745e-01 3.11692698898602960078108026209520e-02 7.77399334956648813133028852462303e-01 4.78204552944848284429246731974672e-03
            3.11692698898602960078108026209520e-02 7.77399334956648813133028852462303e-01 1.91431395153490890859160344916745e-01 4.78204552944848284429246731974672e-03
            7.77399334956648813133028852462303e-01 1.91431395153490890859160344916745e-01 3.11692698898602960078108026209520e-02 4.78204552944848284429246731974672e-03
            1.05850984567690802745643097182437e-02 1.91864903048277385400055550235265e-01 7.97549998494953560346232279698597e-01 5.05492016730007057145890314586723e-03
            1.91864903048277385400055550235265e-01 7.97549998494953560346232279698597e-01 1.05850984567690802745643097182437e-02 5.05492016730007057145890314586723e-03
            7.97549998494953560346232279698597e-01 1.05850984567690802745643097182437e-02 1.91864903048277385400055550235265e-01 5.05492016730007057145890314586723e-03
            2.54202264290056406670359478994214e-02 1.05407606480096824319581116924383e-01 8.69172167090897507257807319547283e-01 6.04520613448391707506202052968547e-03
            1.05407606480096824319581116924383e-01 8.69172167090897507257807319547283e-01 2.54202264290056406670359478994214e-02 6.04520613448391707506202052968547e-03
            8.69172167090897507257807319547283e-01 2.54202264290056406670359478994214e-02 1.05407606480096824319581116924383e-01 6.04520613448391707506202052968547e-03
            5.91292919761095991937338567367988e-01 1.21072892452521504308426969487300e-02 3.96599790993651835080413547984790e-01 6.31270244568983042443566233714591e-03
            1.21072892452521504308426969487300e-02 3.96599790993651835080413547984790e-01 5.91292919761095991937338567367988e-01 6.31270244568983042443566233714591e-03
            3.96599790993651835080413547984790e-01 5.91292919761095991937338567367988e-01 1.21072892452521504308426969487300e-02 6.31270244568983042443566233714591e-03
            1.04903656520941024332671531738015e-01 4.33709753919495621943269725306891e-02 8.51725368087109413473001495731296e-01 6.77640548713800017904818062675076e-03
            4.33709753919495621943269725306891e-02 8.51725368087109413473001495731296e-01 1.04903656520941024332671531738015e-01 6.77640548713800017904818062675076e-03
            8.51725368087109413473001495731296e-01 1.04903656520941024332671531738015e-01 4.33709753919495621943269725306891e-02 6.77640548713800017904818062675076e-03
            4.27217446018734792190230109554250e-01 1.66970788554706185691234310297659e-02 5.56085475125794537198942180111771e-01 7.88479242763255247505860268120159e-03
            1.66970788554706185691234310297659e-02 5.56085475125794537198942180111771e-01 4.27217446018734792190230109554250e-01 7.88479242763255247505860268120159e-03
            5.56085475125794537198942180111771e-01 4.27217446018734792190230109554250e-01 1.66970788554706185691234310297659e-02 7.88479242763255247505860268120159e-03
            1.77899296750879898354469332844019e-01 6.45110680237001715697786607961461e-02 7.57589635225419888442388582916465e-01 8.73690764074747973355616892376929e-03
            6.45110680237001715697786607961461e-02 7.57589635225419888442388582916465e-01 1.77899296750879898354469332844019e-01 8.73690764074747973355616892376929e-03
            7.57589635225419888442388582916465e-01 1.77899296750879898354469332844019e-01 6.45110680237001715697786607961461e-02 8.73690764074747973355616892376929e-03
            8.05482155825715390173868968304305e-02 1.01976993969515816740312175170402e-01 8.17474790447912713631239967071451e-01 9.43440585399333946081679158623956e-03
            1.01976993969515816740312175170402e-01 8.17474790447912713631239967071451e-01 8.05482155825715390173868968304305e-02 9.43440585399333946081679158623956e-03
            8.17474790447912713631239967071451e-01 8.05482155825715390173868968304305e-02 1.01976993969515816740312175170402e-01 9.43440585399333946081679158623956e-03
            3.13173112798550493107718750707136e-02 2.85342978045655892849197243776871e-01 6.83339710674489064778924785059644e-01 9.49087942781928067814689597980760e-03
            2.85342978045655892849197243776871e-01 6.83339710674489064778924785059644e-01 3.13173112798550493107718750707136e-02 9.49087942781928067814689597980760e-03
            6.83339710674489064778924785059644e-01 3.13173112798550493107718750707136e-02 2.85342978045655892849197243776871e-01 9.49087942781928067814689597980760e-03
            2.95446758742554271837121859789477e-01 3.28810125354330601288310731433739e-02 6.71672228722012709667410490510520e-01 1.02109626562177507880235793891188e-02
            3.28810125354330601288310731433739e-02 6.71672228722012709667410490510520e-01 2.95446758742554271837121859789477e-01 1.02109626562177507880235793891188e-02
            6.71672228722012709667410490510520e-01 2.95446758742554271837121859789477e-01 3.28810125354330601288310731433739e-02 1.02109626562177507880235793891188e-02
            5.41132173539479316426437094378343e-02 1.83859582702943807897000283446687e-01 7.62027199943108302093719430558849e-01 1.05192533354850435017846521645879e-02
            1.83859582702943807897000283446687e-01 7.62027199943108302093719430558849e-01 5.41132173539479316426437094378343e-02 1.05192533354850435017846521645879e-02
            7.62027199943108302093719430558849e-01 5.41132173539479316426437094378343e-02 1.83859582702943807897000283446687e-01 1.05192533354850435017846521645879e-02
            4.27400654280490135961123598917766e-02 4.43558986603905025525307337375125e-01 5.13700947968045995573049822269240e-01 1.16891203752775181545775495806083e-02
            4.43558986603905025525307337375125e-01 5.13700947968045995573049822269240e-01 4.27400654280490135961123598917766e-02 1.16891203752775181545775495806083e-02
            5.13700947968045995573049822269240e-01 4.27400654280490135961123598917766e-02 4.43558986603905025525307337375125e-01 1.16891203752775181545775495806083e-02
            1.45187675973377866567304295131180e-01 1.29795450451007293724003943680145e-01 7.25016873575614839708691761188675e-01 1.37004002794014461547034144928148e-02
            1.29795450451007293724003943680145e-01 7.25016873575614839708691761188675e-01 1.45187675973377866567304295131180e-01 1.37004002794014461547034144928148e-02
            7.25016873575614839708691761188675e-01 1.45187675973377866567304295131180e-01 1.29795450451007293724003943680145e-01 1.37004002794014461547034144928148e-02
            7.47139784787638483454230708957766e-02 3.29913961425314705699918249592884e-01 5.95372060095921473710234295140253e-01 1.44604512990827559909545740879366e-02
            3.29913961425314705699918249592884e-01 5.95372060095921473710234295140253e-01 7.47139784787638483454230708957766e-02 1.44604512990827559909545740879366e-02
            5.95372060095921473710234295140253e-01 7.47139784787638483454230708957766e-02 3.29913961425314705699918249592884e-01 1.44604512990827559909545740879366e-02
            3.81968043130448131616816453970387e-01 6.73625275151304159226839374241536e-02 5.50669429354421424704923992976546e-01 1.52313957835889229180947879171981e-02
            6.73625275151304159226839374241536e-02 5.50669429354421424704923992976546e-01 3.81968043130448131616816453970387e-01 1.52313957835889229180947879171981e-02
            5.50669429354421424704923992976546e-01 3.81968043130448131616816453970387e-01 6.73625275151304159226839374241536e-02 1.52313957835889229180947879171981e-02
            2.21783549821814751634718732020701e-01 6.68980984358149566304518884862773e-01 1.09235465820035626549611151858699e-01 1.54858733194458367959933653423832e-02
            6.68980984358149566304518884862773e-01 1.09235465820035626549611151858699e-01 2.21783549821814751634718732020701e-01 1.54858733194458367959933653423832e-02
            1.09235465820035626549611151858699e-01 2.21783549821814751634718732020701e-01 6.68980984358149566304518884862773e-01 1.54858733194458367959933653423832e-02
            2.56870811513146835025622749526519e-01 9.60491211353747958989757194103731e-02 6.47080067351478382953189338877564e-01 1.59889574365544989698051381310506e-02
            9.60491211353747958989757194103731e-02 6.47080067351478382953189338877564e-01 2.56870811513146835025622749526519e-01 1.59889574365544989698051381310506e-02
            6.47080067351478382953189338877564e-01 2.56870811513146835025622749526519e-01 9.60491211353747958989757194103731e-02 1.59889574365544989698051381310506e-02
            1.12751303426959964548714765442128e-01 4.23674182024581569283583348806133e-01 4.63574514548458493923277501380653e-01 1.89290359058916467238642411530236e-02
            4.23674182024581569283583348806133e-01 4.63574514548458493923277501380653e-01 1.12751303426959964548714765442128e-01 1.89290359058916467238642411530236e-02
            4.63574514548458493923277501380653e-01 1.12751303426959964548714765442128e-01 4.23674182024581569283583348806133e-01 1.89290359058916467238642411530236e-02
            2.10323057441033522962925417232327e-01 1.87001109672332688216300766725908e-01 6.02675832886633733309622584783938e-01 1.97480813567340596215160530846333e-02
            1.87001109672332688216300766725908e-01 6.02675832886633733309622584783938e-01 2.10323057441033522962925417232327e-01 1.97480813567340596215160530846333e-02
            6.02675832886633733309622584783938e-01 2.10323057441033522962925417232327e-01 1.87001109672332688216300766725908e-01 1.97480813567340596215160530846333e-02
            1.61575209179476625420690538703639e-01 2.99012948455699423266906933349674e-01 5.39411842364823979067978143575601e-01 2.06677815596659590624550872917098e-02
            2.99012948455699423266906933349674e-01 5.39411842364823979067978143575601e-01 1.61575209179476625420690538703639e-01 2.06677815596659590624550872917098e-02
            5.39411842364823979067978143575601e-01 1.61575209179476625420690538703639e-01 2.99012948455699423266906933349674e-01 2.06677815596659590624550872917098e-02
            3.31656221355574820641720634739613e-01 1.50616235510018708909640849924472e-01 5.17727543134406498204214130964829e-01 2.07983426647951052945728633858380e-02
            1.50616235510018708909640849924472e-01 5.17727543134406498204214130964829e-01 3.31656221355574820641720634739613e-01 2.07983426647951052945728633858380e-02
            5.17727543134406498204214130964829e-01 3.31656221355574820641720634739613e-01 1.50616235510018708909640849924472e-01 2.07983426647951052945728633858380e-02
            2.12294763316891205162662004113372e-01 3.84271485252512368102628670385457e-01 4.03433751430596398979133709872258e-01 2.35755748407441359515424039727804e-02
            3.84271485252512368102628670385457e-01 4.03433751430596398979133709872258e-01 2.12294763316891205162662004113372e-01 2.35755748407441359515424039727804e-02
            4.03433751430596398979133709872258e-01 2.12294763316891205162662004113372e-01 3.84271485252512368102628670385457e-01 2.35755748407441359515424039727804e-02
            2.74113742189350695355187781387940e-01 2.58726723002178027321917852532351e-01 4.67159534808471277322894366079709e-01 2.36918543087045081751984554330193e-02
            2.58726723002178027321917852532351e-01 4.67159534808471277322894366079709e-01 2.74113742189350695355187781387940e-01 2.36918543087045081751984554330193e-02
            4.67159534808471277322894366079709e-01 2.74113742189350695355187781387940e-01 2.58726723002178027321917852532351e-01 2.36918543087045081751984554330193e-02
            ];
        
    case  24
        
        % ALG. DEG.:   24
        % PTS CARD.:  109
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 7.8e-16
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333425851918718763045e-01 1.06817769333868110154250885557303e-02
            7.27744218228642408829998089458968e-03 1.87178063161081678807129424768618e-03 9.90850777186102704696679666085402e-01 3.43072138434039877394332584970016e-04
            1.87178063161081678807129424768618e-03 9.90850777186102704696679666085402e-01 7.27744218228642408829998089458968e-03 3.43072138434039877394332584970016e-04
            9.90850777186102704696679666085402e-01 7.27744218228642408829998089458968e-03 1.87178063161081678807129424768618e-03 3.43072138434039877394332584970016e-04
            9.37621417401595796947688654654485e-02 2.22584177630761334901188241985892e-03 9.04012016483532798716282741224859e-01 1.22299970203740672805570177672507e-03
            2.22584177630761334901188241985892e-03 9.04012016483532798716282741224859e-01 9.37621417401595796947688654654485e-02 1.22299970203740672805570177672507e-03
            9.04012016483532798716282741224859e-01 9.37621417401595796947688654654485e-02 2.22584177630761334901188241985892e-03 1.22299970203740672805570177672507e-03
            2.75499567254815017069979887764930e-02 9.64825831627800223344593177898787e-01 7.62421164671822637615150597412139e-03 1.63816027364919319930791630213207e-03
            9.64825831627800223344593177898787e-01 7.62421164671822637615150597412139e-03 2.75499567254815017069979887764930e-02 1.63816027364919319930791630213207e-03
            7.62421164671822637615150597412139e-03 2.75499567254815017069979887764930e-02 9.64825831627800223344593177898787e-01 1.63816027364919319930791630213207e-03
            3.88491565765760812900531107061397e-02 9.99221468468047303856938157196055e-03 9.51158628738743461283888791513164e-01 2.23722594587012334893527665258262e-03
            9.99221468468047303856938157196055e-03 9.51158628738743461283888791513164e-01 3.88491565765760812900531107061397e-02 2.23722594587012334893527665258262e-03
            9.51158628738743461283888791513164e-01 3.88491565765760812900531107061397e-02 9.99221468468047303856938157196055e-03 2.23722594587012334893527665258262e-03
            8.45824643422994626451583144444157e-02 9.07583350374654984626943132752785e-01 7.83418528304558048347416843171231e-03 2.85750186827827190247375277465380e-03
            9.07583350374654984626943132752785e-01 7.83418528304558048347416843171231e-03 8.45824643422994626451583144444157e-02 2.85750186827827190247375277465380e-03
            7.83418528304558048347416843171231e-03 8.45824643422994626451583144444157e-02 9.07583350374654984626943132752785e-01 2.85750186827827190247375277465380e-03
            1.69711974261767234439801654843905e-01 8.55094769491029395303005600226243e-03 8.21737078043322521914149092481239e-01 3.68582961632744551816465161664382e-03
            8.55094769491029395303005600226243e-03 8.21737078043322521914149092481239e-01 1.69711974261767234439801654843905e-01 3.68582961632744551816465161664382e-03
            8.21737078043322521914149092481239e-01 1.69711974261767234439801654843905e-01 8.55094769491029395303005600226243e-03 3.68582961632744551816465161664382e-03
            2.61114259888929389763490007680957e-01 7.35523531115195392665606632931485e-03 7.31530504799918657177215663978131e-01 3.69479544205243140217453934326386e-03
            7.35523531115195392665606632931485e-03 7.31530504799918657177215663978131e-01 2.61114259888929389763490007680957e-01 3.69479544205243140217453934326386e-03
            7.31530504799918657177215663978131e-01 2.61114259888929389763490007680957e-01 7.35523531115195392665606632931485e-03 3.69479544205243140217453934326386e-03
            1.67919705131083785154544329998316e-01 8.24005083539038452045133453793824e-01 8.07521132987776280032221620786004e-03 3.90203200620795401243334765695181e-03
            8.24005083539038452045133453793824e-01 8.07521132987776280032221620786004e-03 1.67919705131083785154544329998316e-01 3.90203200620795401243334765695181e-03
            8.07521132987776280032221620786004e-03 1.67919705131083785154544329998316e-01 8.24005083539038452045133453793824e-01 3.90203200620795401243334765695181e-03
            3.66562445874964837333465084157069e-01 7.00566016949978002253951459010750e-03 6.26431893955535445961402274406282e-01 4.02806909693100186065084500341982e-03
            7.00566016949978002253951459010750e-03 6.26431893955535445961402274406282e-01 3.66562445874964837333465084157069e-01 4.02806909693100186065084500341982e-03
            6.26431893955535445961402274406282e-01 3.66562445874964837333465084157069e-01 7.00566016949978002253951459010750e-03 4.02806909693100186065084500341982e-03
            2.71431359408183414672066646744497e-01 7.20628297746085721975362048397074e-01 7.94034284573086335257130485842936e-03 4.52488072016167704181288655718163e-03
            7.20628297746085721975362048397074e-01 7.94034284573086335257130485842936e-03 2.71431359408183414672066646744497e-01 4.52488072016167704181288655718163e-03
            7.94034284573086335257130485842936e-03 2.71431359408183414672066646744497e-01 7.20628297746085721975362048397074e-01 4.52488072016167704181288655718163e-03
            4.83415122297192079159344757499639e-01 7.36119356684007599550545464239804e-03 5.09223684135967857855575857684016e-01 4.53960292626740761351422648317566e-03
            7.36119356684007599550545464239804e-03 5.09223684135967857855575857684016e-01 4.83415122297192079159344757499639e-01 4.53960292626740761351422648317566e-03
            5.09223684135967857855575857684016e-01 4.83415122297192079159344757499639e-01 7.36119356684007599550545464239804e-03 4.53960292626740761351422648317566e-03
            3.87957374245691544167868869408267e-01 6.04329908418651196022608473867876e-01 7.71271733565725980952265672385693e-03 4.74569544731183023261733566755538e-03
            6.04329908418651196022608473867876e-01 7.71271733565725980952265672385693e-03 3.87957374245691544167868869408267e-01 4.74569544731183023261733566755538e-03
            7.71271733565725980952265672385693e-03 3.87957374245691544167868869408267e-01 6.04329908418651196022608473867876e-01 4.74569544731183023261733566755538e-03
            5.20536116012078262094142644400563e-02 9.07672123610022807405073308473220e-01 4.02742647887693872021941388084088e-02 5.18053657337445849773160944096162e-03
            9.07672123610022807405073308473220e-01 4.02742647887693872021941388084088e-02 5.20536116012078262094142644400563e-02 5.18053657337445849773160944096162e-03
            4.02742647887693872021941388084088e-02 5.20536116012078262094142644400563e-02 9.07672123610022807405073308473220e-01 5.18053657337445849773160944096162e-03
            2.29018909213725946338868766360974e-01 8.93452564981080527006795932720706e-02 6.81635834288165987082663832552498e-01 5.31192323272322488442176435796682e-03
            8.93452564981080527006795932720706e-02 6.81635834288165987082663832552498e-01 2.29018909213725946338868766360974e-01 5.31192323272322488442176435796682e-03
            6.81635834288165987082663832552498e-01 2.29018909213725946338868766360974e-01 8.93452564981080527006795932720706e-02 5.31192323272322488442176435796682e-03
            9.49141955716360924411034716285940e-02 2.67967190917923619997154105476511e-02 8.78289085336571573314756733452668e-01 5.36419504296679373889222830484869e-03
            2.67967190917923619997154105476511e-02 8.78289085336571573314756733452668e-01 9.49141955716360924411034716285940e-02 5.36419504296679373889222830484869e-03
            8.78289085336571573314756733452668e-01 9.49141955716360924411034716285940e-02 2.67967190917923619997154105476511e-02 5.36419504296679373889222830484869e-03
            1.23228014736413052365726628067932e-01 8.35184452859199955021551886602538e-01 4.15875324043869509793580618861597e-02 7.84096594518897405112767273749341e-03
            8.35184452859199955021551886602538e-01 4.15875324043869509793580618861597e-02 1.23228014736413052365726628067932e-01 7.84096594518897405112767273749341e-03
            4.15875324043869509793580618861597e-02 1.23228014736413052365726628067932e-01 8.35184452859199955021551886602538e-01 7.84096594518897405112767273749341e-03
            1.74126507274859759766272304659651e-01 4.41306288863040421799333046237734e-02 7.81742863838836177237112678994890e-01 8.69940226621333627388121811918609e-03
            4.41306288863040421799333046237734e-02 7.81742863838836177237112678994890e-01 1.74126507274859759766272304659651e-01 8.69940226621333627388121811918609e-03
            7.81742863838836177237112678994890e-01 1.74126507274859759766272304659651e-01 4.41306288863040421799333046237734e-02 8.69940226621333627388121811918609e-03
            2.72180746005686646782351090223528e-01 3.74687219394429782415301133369212e-02 6.90350532054870402731694412068464e-01 8.84020363336787337593669633406535e-03
            3.74687219394429782415301133369212e-02 6.90350532054870402731694412068464e-01 2.72180746005686646782351090223528e-01 8.84020363336787337593669633406535e-03
            6.90350532054870402731694412068464e-01 2.72180746005686646782351090223528e-01 3.74687219394429782415301133369212e-02 8.84020363336787337593669633406535e-03
            9.90063515075153971478627568103548e-02 8.19687967813020335050566700374475e-02 8.19024851711182666491595227853395e-01 9.62913221988995898792396133103466e-03
            8.19687967813020335050566700374475e-02 8.19024851711182666491595227853395e-01 9.90063515075153971478627568103548e-02 9.62913221988995898792396133103466e-03
            8.19024851711182666491595227853395e-01 9.90063515075153971478627568103548e-02 8.19687967813020335050566700374475e-02 9.62913221988995898792396133103466e-03
            2.18932787550637497631100814032834e-01 7.39314412421514188800131250900449e-01 4.17528000278483135687679350667167e-02 9.80325260988398027028889458733829e-03
            7.39314412421514188800131250900449e-01 4.17528000278483135687679350667167e-02 2.18932787550637497631100814032834e-01 9.80325260988398027028889458733829e-03
            4.17528000278483135687679350667167e-02 2.18932787550637497631100814032834e-01 7.39314412421514188800131250900449e-01 9.80325260988398027028889458733829e-03
            3.85457856195294268353279676375678e-01 3.70114098745062680695383505735663e-02 5.77530733930199491332757588679669e-01 9.94281737582886245008229053610194e-03
            3.70114098745062680695383505735663e-02 5.77530733930199491332757588679669e-01 3.85457856195294268353279676375678e-01 9.94281737582886245008229053610194e-03
            5.77530733930199491332757588679669e-01 3.85457856195294268353279676375678e-01 3.70114098745062680695383505735663e-02 9.94281737582886245008229053610194e-03
            4.53931532973698426136621719706454e-01 5.07124120567042480089980927004945e-01 3.89443464592591492845485845464282e-02 1.07482407367000144110535941877060e-02
            5.07124120567042480089980927004945e-01 3.89443464592591492845485845464282e-02 4.53931532973698426136621719706454e-01 1.07482407367000144110535941877060e-02
            3.89443464592591492845485845464282e-02 4.53931532973698426136621719706454e-01 5.07124120567042480089980927004945e-01 1.07482407367000144110535941877060e-02
            3.31414067992492400449577871768270e-01 6.27872100319265880052910233644070e-01 4.07138316882417194975118945876602e-02 1.07752726636044619812349054654987e-02
            6.27872100319265880052910233644070e-01 4.07138316882417194975118945876602e-02 3.31414067992492400449577871768270e-01 1.07752726636044619812349054654987e-02
            4.07138316882417194975118945876602e-02 3.31414067992492400449577871768270e-01 6.27872100319265880052910233644070e-01 1.07752726636044619812349054654987e-02
            1.75418478700220814570087668471388e-01 1.09788999485381039145082127106434e-01 7.14792521814398118529254588793265e-01 1.17830977528251903962974722617219e-02
            1.09788999485381039145082127106434e-01 7.14792521814398118529254588793265e-01 1.75418478700220814570087668471388e-01 1.17830977528251903962974722617219e-02
            7.14792521814398118529254588793265e-01 1.75418478700220814570087668471388e-01 1.09788999485381039145082127106434e-01 1.17830977528251903962974722617219e-02
            2.99424606258227421484718888677889e-01 8.95177416056505692587563771667192e-02 6.11057652136121953745373502897564e-01 1.21567676541047674926288024721543e-02
            8.95177416056505692587563771667192e-02 6.11057652136121953745373502897564e-01 2.99424606258227421484718888677889e-01 1.21567676541047674926288024721543e-02
            6.11057652136121953745373502897564e-01 2.99424606258227421484718888677889e-01 8.95177416056505692587563771667192e-02 1.21567676541047674926288024721543e-02
            1.64438284214270008520841770405241e-01 7.35720413124063998822066423599608e-01 9.98413026616660204126674216240644e-02 1.29457052361328234235671175156313e-02
            7.35720413124063998822066423599608e-01 9.98413026616660204126674216240644e-02 1.64438284214270008520841770405241e-01 1.29457052361328234235671175156313e-02
            9.98413026616660204126674216240644e-02 1.64438284214270008520841770405241e-01 7.35720413124063998822066423599608e-01 1.29457052361328234235671175156313e-02
            3.69724758939286801684431793546537e-01 4.63127807453672823978507722131326e-01 1.67147433607040374337060484322137e-01 1.35548997474791085848799809809861e-02
            4.63127807453672823978507722131326e-01 1.67147433607040374337060484322137e-01 3.69724758939286801684431793546537e-01 1.35548997474791085848799809809861e-02
            1.67147433607040374337060484322137e-01 3.69724758939286801684431793546537e-01 4.63127807453672823978507722131326e-01 1.35548997474791085848799809809861e-02
            4.06369061777283435699104074956267e-01 9.07167861113062495848069488602050e-02 5.02914152111410217571574321482331e-01 1.46758324930695502330557644654618e-02
            9.07167861113062495848069488602050e-02 5.02914152111410217571574321482331e-01 4.06369061777283435699104074956267e-01 1.46758324930695502330557644654618e-02
            5.02914152111410217571574321482331e-01 4.06369061777283435699104074956267e-01 9.07167861113062495848069488602050e-02 1.46758324930695502330557644654618e-02
            2.67420664299267585306552064139396e-01 6.32902725169850244313352050085086e-01 9.96766105308821703800958857755177e-02 1.54204544583998980067374162672422e-02
            6.32902725169850244313352050085086e-01 9.96766105308821703800958857755177e-02 2.67420664299267585306552064139396e-01 1.54204544583998980067374162672422e-02
            9.96766105308821703800958857755177e-02 2.67420664299267585306552064139396e-01 6.32902725169850244313352050085086e-01 1.54204544583998980067374162672422e-02
            3.83204107909751179317225933118607e-01 5.21623720050690464766773857263615e-01 9.51721720395583004048489783599507e-02 1.55159977941551948527054705095907e-02
            5.21623720050690464766773857263615e-01 9.51721720395583004048489783599507e-02 3.83204107909751179317225933118607e-01 1.55159977941551948527054705095907e-02
            9.51721720395583004048489783599507e-02 3.83204107909751179317225933118607e-01 5.21623720050690464766773857263615e-01 1.55159977941551948527054705095907e-02
            1.78724492320244954246533097830252e-01 1.92031135757774140682130337154376e-01 6.29244371921980905071336565015372e-01 1.56625893060099094511450346089987e-02
            1.92031135757774140682130337154376e-01 6.29244371921980905071336565015372e-01 1.78724492320244954246533097830252e-01 1.56625893060099094511450346089987e-02
            6.29244371921980905071336565015372e-01 1.78724492320244954246533097830252e-01 1.92031135757774140682130337154376e-01 1.56625893060099094511450346089987e-02
            2.82829128593355183518553985777544e-01 5.37642437490313485071169452567119e-01 1.79528433916331331410276561655337e-01 1.59275375108478001173750726593425e-02
            5.37642437490313485071169452567119e-01 1.79528433916331331410276561655337e-01 2.82829128593355183518553985777544e-01 1.59275375108478001173750726593425e-02
            1.79528433916331331410276561655337e-01 2.82829128593355183518553985777544e-01 5.37642437490313485071169452567119e-01 1.59275375108478001173750726593425e-02
            2.70606479254229259634456639105338e-01 1.61339950052654446110977914941031e-01 5.68053570693116238743414214695804e-01 1.61035330300796433733445667257911e-02
            1.61339950052654446110977914941031e-01 5.68053570693116238743414214695804e-01 2.70606479254229259634456639105338e-01 1.61035330300796433733445667257911e-02
            5.68053570693116238743414214695804e-01 2.70606479254229259634456639105338e-01 1.61339950052654446110977914941031e-01 1.61035330300796433733445667257911e-02
            3.70874346744923255592141231318237e-01 1.64548573538222442369516329563339e-01 4.64577079716854302038342439118423e-01 1.66580804257312579952365894087052e-02
            1.64548573538222442369516329563339e-01 4.64577079716854302038342439118423e-01 3.70874346744923255592141231318237e-01 1.66580804257312579952365894087052e-02
            4.64577079716854302038342439118423e-01 3.70874346744923255592141231318237e-01 1.64548573538222442369516329563339e-01 1.66580804257312579952365894087052e-02
            3.63840811889283188840238381089875e-01 3.83367191770047632815021643182263e-01 2.52791996340669178344739975727862e-01 1.93530794923629304027379305352952e-02
            3.83367191770047632815021643182263e-01 2.52791996340669178344739975727862e-01 3.63840811889283188840238381089875e-01 1.93530794923629304027379305352952e-02
            2.52791996340669178344739975727862e-01 3.63840811889283188840238381089875e-01 3.83367191770047632815021643182263e-01 1.93530794923629304027379305352952e-02
            2.75735321862771332046548877769965e-01 2.52760449512089535595293909864267e-01 4.71504228625139076847005981107941e-01 2.04593586377354516581128507368703e-02
            2.52760449512089535595293909864267e-01 4.71504228625139076847005981107941e-01 2.75735321862771332046548877769965e-01 2.04593586377354516581128507368703e-02
            4.71504228625139076847005981107941e-01 2.75735321862771332046548877769965e-01 2.52760449512089535595293909864267e-01 2.04593586377354516581128507368703e-02
            ];
        
    case  25
        
        % ALG. DEG.:   25
        % PTS CARD.:  117
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.4e-16
        
        xyw_bar=[
            1.31881771474051147363670821732740e-02 3.07956384011914569703849409165741e-03 9.83732259012475740433956161723472e-01 5.26947810971040804181586558740946e-04
            3.07956384011914569703849409165741e-03 9.83732259012475740433956161723472e-01 1.31881771474051147363670821732740e-02 5.26947810971040804181586558740946e-04
            9.83732259012475740433956161723472e-01 1.31881771474051147363670821732740e-02 3.07956384011914569703849409165741e-03 5.26947810971040804181586558740946e-04
            2.72596213613051892646343965509459e-03 1.71043047595741092015142470472711e-02 9.80169733104295337611233662755694e-01 5.38378571248994018325073618314036e-04
            1.71043047595741092015142470472711e-02 9.80169733104295337611233662755694e-01 2.72596213613051892646343965509459e-03 5.38378571248994018325073618314036e-04
            9.80169733104295337611233662755694e-01 2.72596213613051892646343965509459e-03 1.71043047595741092015142470472711e-02 5.38378571248994018325073618314036e-04
            6.29840020924785359834885412055883e-02 9.31371984168810507220825911645079e-01 5.64401373871092904010993152041920e-03 1.71255729388201764773480562098484e-03
            9.31371984168810507220825911645079e-01 5.64401373871092904010993152041920e-03 6.29840020924785359834885412055883e-02 1.71255729388201764773480562098484e-03
            5.64401373871092904010993152041920e-03 6.29840020924785359834885412055883e-02 9.31371984168810507220825911645079e-01 1.71255729388201764773480562098484e-03
            5.20238006731535140847455522816745e-02 7.43791496086179931140902610309240e-03 9.40538284365984744717081866838271e-01 1.91233286697058395969817112813871e-03
            7.43791496086179931140902610309240e-03 9.40538284365984744717081866838271e-01 5.20238006731535140847455522816745e-02 1.91233286697058395969817112813871e-03
            9.40538284365984744717081866838271e-01 5.20238006731535140847455522816745e-02 7.43791496086179931140902610309240e-03 1.91233286697058395969817112813871e-03
            2.20200206141671100956713758023398e-01 7.76414334424490371056037929520244e-01 3.38545943383850023167269682744518e-03 1.98974451400670687695182436982577e-03
            7.76414334424490371056037929520244e-01 3.38545943383850023167269682744518e-03 2.20200206141671100956713758023398e-01 1.98974451400670687695182436982577e-03
            3.38545943383850023167269682744518e-03 2.20200206141671100956713758023398e-01 7.76414334424490371056037929520244e-01 1.98974451400670687695182436982577e-03
            3.04395561568152825393518412511185e-02 9.44856451527789409361446359980619e-01 2.47039923153953289158835104899481e-02 2.70300175161245935170906129485502e-03
            9.44856451527789409361446359980619e-01 2.47039923153953289158835104899481e-02 3.04395561568152825393518412511185e-02 2.70300175161245935170906129485502e-03
            2.47039923153953289158835104899481e-02 3.04395561568152825393518412511185e-02 9.44856451527789409361446359980619e-01 2.70300175161245935170906129485502e-03
            1.13993174377526096385793152876431e-01 7.47102931577944221613885034116720e-03 8.78535796306694427570960215234663e-01 2.82376214206483521634849509496235e-03
            7.47102931577944221613885034116720e-03 8.78535796306694427570960215234663e-01 1.13993174377526096385793152876431e-01 2.82376214206483521634849509496235e-03
            8.78535796306694427570960215234663e-01 1.13993174377526096385793152876431e-01 7.47102931577944221613885034116720e-03 2.82376214206483521634849509496235e-03
            3.42295210188022336339486173528712e-01 6.52613563397589468983994720474584e-01 5.09122641438819467651910599670373e-03 2.91514917247315307261001926519839e-03
            6.52613563397589468983994720474584e-01 5.09122641438819467651910599670373e-03 3.42295210188022336339486173528712e-01 2.91514917247315307261001926519839e-03
            5.09122641438819467651910599670373e-03 3.42295210188022336339486173528712e-01 6.52613563397589468983994720474584e-01 2.91514917247315307261001926519839e-03
            1.30473991950559503605333588893700e-01 8.61524966076299802963944785005879e-01 8.00104197314066567514601047150791e-03 3.14622878230012923556069281971759e-03
            8.61524966076299802963944785005879e-01 8.00104197314066567514601047150791e-03 1.30473991950559503605333588893700e-01 3.14622878230012923556069281971759e-03
            8.00104197314066567514601047150791e-03 1.30473991950559503605333588893700e-01 8.61524966076299802963944785005879e-01 3.14622878230012923556069281971759e-03
            2.00179308343867129460491582904069e-01 6.50781868302953165478230701523898e-03 7.93312872973103377916004319558851e-01 3.25439774886752226576436264338099e-03
            6.50781868302953165478230701523898e-03 7.93312872973103377916004319558851e-01 2.00179308343867129460491582904069e-01 3.25439774886752226576436264338099e-03
            7.93312872973103377916004319558851e-01 2.00179308343867129460491582904069e-01 6.50781868302953165478230701523898e-03 3.25439774886752226576436264338099e-03
            3.07372363215433874739090924776974e-01 6.35240709459219433929977327579763e-03 6.86275229689973897961863258387893e-01 3.72306578149784973483238381675164e-03
            6.35240709459219433929977327579763e-03 6.86275229689973897961863258387893e-01 3.07372363215433874739090924776974e-01 3.72306578149784973483238381675164e-03
            6.86275229689973897961863258387893e-01 3.07372363215433874739090924776974e-01 6.35240709459219433929977327579763e-03 3.72306578149784973483238381675164e-03
            2.76217326622710424022244524167036e-01 7.06370557783410402130641614348860e-01 1.74121155938792293582650927419309e-02 4.30631473928152954527837081855068e-03
            7.06370557783410402130641614348860e-01 1.74121155938792293582650927419309e-02 2.76217326622710424022244524167036e-01 4.30631473928152954527837081855068e-03
            1.74121155938792293582650927419309e-02 2.76217326622710424022244524167036e-01 7.06370557783410402130641614348860e-01 4.30631473928152954527837081855068e-03
            4.22935483077426899178874464269029e-01 7.56275469533758924745026064329068e-03 5.69501762227235475144482279574731e-01 4.52513415896231303153784963910766e-03
            7.56275469533758924745026064329068e-03 5.69501762227235475144482279574731e-01 4.22935483077426899178874464269029e-01 4.52513415896231303153784963910766e-03
            5.69501762227235475144482279574731e-01 4.22935483077426899178874464269029e-01 7.56275469533758924745026064329068e-03 4.52513415896231303153784963910766e-03
            4.53227003366671432615930825704709e-01 5.38998045885384025055486745259259e-01 7.77495074794454232858242903603241e-03 4.56249922131780807216250295255122e-03
            5.38998045885384025055486745259259e-01 7.77495074794454232858242903603241e-03 4.53227003366671432615930825704709e-01 4.56249922131780807216250295255122e-03
            7.77495074794454232858242903603241e-03 4.53227003366671432615930825704709e-01 5.38998045885384025055486745259259e-01 4.56249922131780807216250295255122e-03
            1.90195202693624082490586602034455e-01 7.84036511910398581370884585339809e-01 2.57682853959773083829531969968230e-02 4.68027662060985401304424868840215e-03
            7.84036511910398581370884585339809e-01 2.57682853959773083829531969968230e-02 1.90195202693624082490586602034455e-01 4.68027662060985401304424868840215e-03
            2.57682853959773083829531969968230e-02 1.90195202693624082490586602034455e-01 7.84036511910398581370884585339809e-01 4.68027662060985401304424868840215e-03
            3.61718299806125034856485456202790e-02 8.44564005292429142990329182794085e-02 8.79371769490144616909788055636454e-01 5.64055598154384295328522824775064e-03
            8.44564005292429142990329182794085e-02 8.79371769490144616909788055636454e-01 3.61718299806125034856485456202790e-02 5.64055598154384295328522824775064e-03
            8.79371769490144616909788055636454e-01 3.61718299806125034856485456202790e-02 8.44564005292429142990329182794085e-02 5.64055598154384295328522824775064e-03
            7.60794388659036535660362687849556e-02 4.05076326129555214228261661446595e-02 8.83412928521140838888925372884842e-01 5.77842305077480751507312817238926e-03
            4.05076326129555214228261661446595e-02 8.83412928521140838888925372884842e-01 7.60794388659036535660362687849556e-02 5.77842305077480751507312817238926e-03
            8.83412928521140838888925372884842e-01 7.60794388659036535660362687849556e-02 4.05076326129555214228261661446595e-02 5.77842305077480751507312817238926e-03
            1.49945977487857845522611910382693e-01 3.58350589330340557614462682067824e-02 8.14218963579108057082578397967154e-01 6.57914135071524693032518626978344e-03
            3.58350589330340557614462682067824e-02 8.14218963579108057082578397967154e-01 1.49945977487857845522611910382693e-01 6.57914135071524693032518626978344e-03
            8.14218963579108057082578397967154e-01 1.49945977487857845522611910382693e-01 3.58350589330340557614462682067824e-02 6.57914135071524693032518626978344e-03
            1.50955341179002944240750139215379e-01 7.96913702682935043419831799837993e-01 5.21309561380619568282668296888005e-02 6.72965316584276176947732039934635e-03
            7.96913702682935043419831799837993e-01 5.21309561380619568282668296888005e-02 1.50955341179002944240750139215379e-01 6.72965316584276176947732039934635e-03
            5.21309561380619568282668296888005e-02 1.50955341179002944240750139215379e-01 7.96913702682935043419831799837993e-01 6.72965316584276176947732039934635e-03
            2.44783310976574935979854785728094e-01 3.26897799308286454711414137364045e-02 7.22526909092596425487897704442730e-01 7.57554243586837025570268622232106e-03
            3.26897799308286454711414137364045e-02 7.22526909092596425487897704442730e-01 2.44783310976574935979854785728094e-01 7.57554243586837025570268622232106e-03
            7.22526909092596425487897704442730e-01 2.44783310976574935979854785728094e-01 3.26897799308286454711414137364045e-02 7.57554243586837025570268622232106e-03
            1.70674433667943242776132706239878e-01 7.87558693381886776307965192245319e-02 7.50569696993868107348646390164504e-01 8.14583504633362848190714089469111e-03
            7.87558693381886776307965192245319e-02 7.50569696993868107348646390164504e-01 1.70674433667943242776132706239878e-01 8.14583504633362848190714089469111e-03
            7.50569696993868107348646390164504e-01 1.70674433667943242776132706239878e-01 7.87558693381886776307965192245319e-02 8.14583504633362848190714089469111e-03
            3.71942611691943147977923445068882e-01 5.93428569491770030452926221187226e-01 3.46288188162868770803015650017187e-02 9.08183523962656329142628663930736e-03
            5.93428569491770030452926221187226e-01 3.46288188162868770803015650017187e-02 3.71942611691943147977923445068882e-01 9.08183523962656329142628663930736e-03
            3.46288188162868770803015650017187e-02 3.71942611691943147977923445068882e-01 5.93428569491770030452926221187226e-01 9.08183523962656329142628663930736e-03
            3.51792010432223745119983959739329e-01 3.63528731441380772992921777131414e-02 6.11855116423638212275193382083671e-01 9.40089530430836088636947778240938e-03
            3.63528731441380772992921777131414e-02 6.11855116423638212275193382083671e-01 3.51792010432223745119983959739329e-01 9.40089530430836088636947778240938e-03
            6.11855116423638212275193382083671e-01 3.51792010432223745119983959739329e-01 3.63528731441380772992921777131414e-02 9.40089530430836088636947778240938e-03
            9.84400611472149716929536111820198e-02 9.69591386055578818847777711198432e-02 8.04600800247227132544480809883680e-01 9.52400454407075283680406130315532e-03
            9.69591386055578818847777711198432e-02 8.04600800247227132544480809883680e-01 9.84400611472149716929536111820198e-02 9.52400454407075283680406130315532e-03
            8.04600800247227132544480809883680e-01 9.84400611472149716929536111820198e-02 9.69591386055578818847777711198432e-02 9.52400454407075283680406130315532e-03
            2.60865026298107871571829718959634e-01 6.84089653184228985161041691753780e-01 5.50453205176631987782798205444124e-02 1.01072081452672178636387201322577e-02
            6.84089653184228985161041691753780e-01 5.50453205176631987782798205444124e-02 2.60865026298107871571829718959634e-01 1.01072081452672178636387201322577e-02
            5.50453205176631987782798205444124e-02 2.60865026298107871571829718959634e-01 6.84089653184228985161041691753780e-01 1.01072081452672178636387201322577e-02
            3.66582144327067316336155045064515e-01 5.50781902551756963326567984040594e-01 8.26359531211756648261257396370638e-02 1.08591268077148413867538678800884e-02
            5.50781902551756963326567984040594e-01 8.26359531211756648261257396370638e-02 3.66582144327067316336155045064515e-01 1.08591268077148413867538678800884e-02
            8.26359531211756648261257396370638e-02 3.66582144327067316336155045064515e-01 5.50781902551756963326567984040594e-01 1.08591268077148413867538678800884e-02
            4.68735987743839010999380434441264e-01 4.11368221710290485337857546710438e-02 4.90127190085131891894576483537094e-01 1.11689841862876882566668612639660e-02
            4.11368221710290485337857546710438e-02 4.90127190085131891894576483537094e-01 4.68735987743839010999380434441264e-01 1.11689841862876882566668612639660e-02
            4.90127190085131891894576483537094e-01 4.68735987743839010999380434441264e-01 4.11368221710290485337857546710438e-02 1.11689841862876882566668612639660e-02
            2.57617931613541240043474545018398e-01 8.24865172853232742067319804846193e-02 6.59895551101135513505369090125896e-01 1.17722968652321513527425622669398e-02
            8.24865172853232742067319804846193e-02 6.59895551101135513505369090125896e-01 2.57617931613541240043474545018398e-01 1.17722968652321513527425622669398e-02
            6.59895551101135513505369090125896e-01 2.57617931613541240043474545018398e-01 8.24865172853232742067319804846193e-02 1.17722968652321513527425622669398e-02
            1.84517952412833835529326620417123e-01 7.12575925803732079799601706326939e-01 1.02906121783434056915496057627024e-01 1.21424040084771537456598267112895e-02
            7.12575925803732079799601706326939e-01 1.02906121783434056915496057627024e-01 1.84517952412833835529326620417123e-01 1.21424040084771537456598267112895e-02
            1.02906121783434056915496057627024e-01 1.84517952412833835529326620417123e-01 7.12575925803732079799601706326939e-01 1.21424040084771537456598267112895e-02
            2.81209118651227718554963530550594e-01 6.05257308197521104453642237785971e-01 1.13533573151251121480243000405608e-01 1.25526989266996867600223453109720e-02
            6.05257308197521104453642237785971e-01 1.13533573151251121480243000405608e-01 2.81209118651227718554963530550594e-01 1.25526989266996867600223453109720e-02
            1.13533573151251121480243000405608e-01 2.81209118651227718554963530550594e-01 6.05257308197521104453642237785971e-01 1.25526989266996867600223453109720e-02
            1.73591491675633341484896732254128e-01 1.42209318699190873092064180127636e-01 6.84199189625175785423039087618235e-01 1.27814412760983882322296523170735e-02
            1.42209318699190873092064180127636e-01 6.84199189625175785423039087618235e-01 1.73591491675633341484896732254128e-01 1.27814412760983882322296523170735e-02
            6.84199189625175785423039087618235e-01 1.73591491675633341484896732254128e-01 1.42209318699190873092064180127636e-01 1.27814412760983882322296523170735e-02
            4.42362027995821582848634534457233e-01 4.54988327657975588635963504202664e-01 1.02649644346202828515401961340103e-01 1.32468413876246516452939161467839e-02
            4.54988327657975588635963504202664e-01 1.02649644346202828515401961340103e-01 4.42362027995821582848634534457233e-01 1.32468413876246516452939161467839e-02
            1.02649644346202828515401961340103e-01 4.42362027995821582848634534457233e-01 4.54988327657975588635963504202664e-01 1.32468413876246516452939161467839e-02
            3.59368226555797587451479557785206e-01 9.07953156997939186956259050020890e-02 5.49836457744408479975106729398249e-01 1.38282415635121516683803122305108e-02
            9.07953156997939186956259050020890e-02 5.49836457744408479975106729398249e-01 3.59368226555797587451479557785206e-01 1.38282415635121516683803122305108e-02
            5.49836457744408479975106729398249e-01 3.59368226555797587451479557785206e-01 9.07953156997939186956259050020890e-02 1.38282415635121516683803122305108e-02
            1.82134268926742365346527208203042e-01 2.28667042140729831478651590259688e-01 5.89198688932527803174821201537270e-01 1.59282121621503146291498609343762e-02
            2.28667042140729831478651590259688e-01 5.89198688932527803174821201537270e-01 1.82134268926742365346527208203042e-01 1.59282121621503146291498609343762e-02
            5.89198688932527803174821201537270e-01 1.82134268926742365346527208203042e-01 2.28667042140729831478651590259688e-01 1.59282121621503146291498609343762e-02
            2.67343042736282066051956007868284e-01 1.56184413818651390837288772672764e-01 5.76472543445066598621906450716779e-01 1.67459571385038495650032785988515e-02
            1.56184413818651390837288772672764e-01 5.76472543445066598621906450716779e-01 2.67343042736282066051956007868284e-01 1.67459571385038495650032785988515e-02
            5.76472543445066598621906450716779e-01 2.67343042736282066051956007868284e-01 1.56184413818651390837288772672764e-01 1.67459571385038495650032785988515e-02
            3.65290320906562715475018876531976e-01 1.73792802318537387140651162553695e-01 4.60916876774899841873178729656502e-01 1.87746105537932306273063431945047e-02
            1.73792802318537387140651162553695e-01 4.60916876774899841873178729656502e-01 3.65290320906562715475018876531976e-01 1.87746105537932306273063431945047e-02
            4.60916876774899841873178729656502e-01 3.65290320906562715475018876531976e-01 1.73792802318537387140651162553695e-01 1.87746105537932306273063431945047e-02
            3.45064826601384877502454173736623e-01 4.82299125389173410916043849283597e-01 1.72636048009441656070350745721953e-01 1.97380050801869745547012513497975e-02
            4.82299125389173410916043849283597e-01 1.72636048009441656070350745721953e-01 3.45064826601384877502454173736623e-01 1.97380050801869745547012513497975e-02
            1.72636048009441656070350745721953e-01 3.45064826601384877502454173736623e-01 4.82299125389173410916043849283597e-01 1.97380050801869745547012513497975e-02
            2.65562227862875310790968796936795e-01 2.56802629703844209618779359516338e-01 4.77635142433280479590251843546866e-01 2.02862843586862802436332486877291e-02
            2.56802629703844209618779359516338e-01 4.77635142433280479590251843546866e-01 2.65562227862875310790968796936795e-01 2.02862843586862802436332486877291e-02
            4.77635142433280479590251843546866e-01 2.65562227862875310790968796936795e-01 2.56802629703844209618779359516338e-01 2.02862843586862802436332486877291e-02
            3.62956385501777445412585620942991e-01 3.71550947373690598762863146475866e-01 2.65492667124531955824551232581143e-01 2.16253435779476220479633497006944e-02
            3.71550947373690598762863146475866e-01 2.65492667124531955824551232581143e-01 3.62956385501777445412585620942991e-01 2.16253435779476220479633497006944e-02
            2.65492667124531955824551232581143e-01 3.62956385501777445412585620942991e-01 3.71550947373690598762863146475866e-01 2.16253435779476220479633497006944e-02
            ];
        
        
    otherwise
        xyw=[];
end




pointset_stats=[
    10   24 3.16e-08    0    0
    11   27 3.16e-08    0    0
    12   31 3.31e-07    0    3
    13   36 3.31e-07    0    0
    14   40 4.97e-07    0    3
    15   46 4.97e-07    0    0
    16   51 2.31e-06    0    6
    17   57 5.83e-16    0    0
    18   64 4.86e-16    1    0
    19   70 4.16e-16    0    0
    20   78 4.89e-16    0    0
    21   85 4.64e-16    0    0
    22   93 7.56e-16    0    0
    23  100 6.66e-16    0    0
    24  109 7.81e-16    0    0
    25  117 5.41e-16    0    0
    ];





