
 	function [xyw_bar,pointset_stats]=set_papanicolopulos_A_IV_C_barycentric(deg)
 	
 	 %--------------------------------------------------------------------
 	 % CUBATURE RULE ON THE SIMPLEX [0 0; 0 1; 1 0] WITH ADE EQUAL TO deg.
 	 % THE CUBATURE RULE IS DESCRIBED IN BARYCENTRICAL FORM.
 	 %--------------------------------------------------------------------
 	 % INPUT: 
 	 % deg: algebraic degree of precision (ADE) of the rule.
 	 %--------------------------------------------------------------------
 	 % OUTPUT: 
 	 % xyw_bar: MATRIX OF NODES xyw(:,1:3) IN BARYCENTRICAL COOR. AND WEIGHTS xyw(:,4).
 	 %      REMARK: The sum of the weights makes 1.
 	 % 
 	 % pointset_stats:
 	 %    statistics of the pointset stored in a matrix. Each rows contains
 	 %    degree, cardinality of the full pointset, moment error w.r.t. 
 	 %    orthn. Dubiner basis, number of points with negative weights,
 	 %    number of points outside the domain.
 	 %--------------------------------------------------------------------
 % SOURCE:
% S-A. Papanicoloupolos
% "New fully symmetric and rotationally symmetric cubature rules on the
% triangle using minimal orthonormal bases".
% JCAM 294 (2016), 39-48.
% ** Using file rotationalsymmetry.txt from archives.
%--------------------------------------------------------------------
% REMARK:
% Rules in the original set not appropriate for cubature when deg <= 16.
%--------------------------------------------------------------------
% | DEG | CARD | W- | PT.O|   M.E.   |Q.|
%--------------------------------------------------------------------
% |  10 |  24 |   0 |   0 | 2.08e-15 |PI|
% |  11 |  27 |   0 |   0 | 9.92e-16 |PI|
% |  12 |  31 |   0 |   3 | 2.21e-15 |P0|
% |  13 |  36 |   0 |   0 | 3.05e-16 |PI|
% |  14 |  40 |   0 |   3 | 9.51e-16 |P0|
% |  15 |  46 |   0 |   0 | 5.00e-16 |PI|
% |  16 |  51 |   0 |   6 | 1.78e-15 |P0|
% |  17 |  57 |   0 |   0 | 5.83e-16 |PI|
% |  18 |  64 |   1 |   0 | 4.86e-16 |NI|
% |  19 |  70 |   0 |   0 | 4.16e-16 |PI|
% |  20 |  78 |   0 |   0 | 4.89e-16 |PI|
% |  21 |  85 |   0 |   0 | 4.64e-16 |PI|
% |  22 |  93 |   0 |   0 | 7.56e-16 |PI|
% |  23 | 100 |   0 |   0 | 6.66e-16 |PI|
% |  24 | 109 |   0 |   0 | 7.81e-16 |PI|
% |  25 | 117 |   0 |   0 | 5.41e-16 |PI|
%--------------------------------------------------------------------
% LEGEND:
%--------------------------------------------------------------------
% DEG  : algebraic degree of precision
% CARD : cardinality of the pointset
% W-   : negative weights
% PT.O : points outside the simplex [0 0; 1 0; 0 1].
% M.E. : maximum moment error in inf norm.
% Q.: quality of the rule.
%        PI: pos. weights, points in simplex
%        NI: some neg. weights, points in simplex
%        P0: pos. weights, some points out of the simplex
%        N0: some neg. weights, some points out of the simplex
%--------------------------------------------------------------------
 
 
 	 switch deg
 
 	 case  10 

 	 % ALG. DEG.:   10
 	 % PTS CARD.:   24
 	 % NEG. W.  :    0
 	 % OUT PTS. :    0
 	 % M.E.INF. : 2.1e-15  

 	 xyw_bar=[
 	 1.45834576989294056881263372815738e-01 1.13985726592417714597704758716645e-02 8.42766850351464147372837487637298e-01 1.47111434032494211437969156008876e-02
 	 1.13985706737416217504410553829075e-02 8.42766851850608622598315378127154e-01 1.45834577475649762590137470397167e-01 1.47111425048221788197677639686844e-02
 	 8.42766841218143714087318585370667e-01 1.45834585819399425998810215787671e-01 1.13985729624568599138711988416617e-02 1.47111439765435590004694077492786e-02
 	 4.80639357548164503075582842939184e-01 3.96436775306336502966431112326973e-03 5.15396274698772138833646749844775e-01 1.49285916217994207161323672039543e-02
 	 3.96437233272861535432474155982163e-03 5.15396271752390666698318000271684e-01 4.80639355914880717079995520180091e-01 1.49285937116037745880925413644036e-02
 	 5.15396271752390666698318000271684e-01 4.80639356407453255393136259954190e-01 3.96437184015607790854573977412656e-03 1.49285941683405358471059543035153e-02
 	 2.83843627688640069706149660078154e-03 2.35341709211809968493867017969023e-01 7.61819854511303629074348009453388e-01 1.49473235787973713489806826260065e-02
 	 2.35341708166612734176581511746917e-01 7.61819851839867112630599876865745e-01 2.83843999352018094839422701625153e-03 1.49473254358686614973183992560735e-02
 	 7.61819857564093094737245337455533e-01 2.83843669030111114826020468626666e-03 2.35341705745605783706153601997357e-01 1.49473237904687323435171109053954e-02
 	 2.73824571185072064560817040046459e-02 4.33992887961679035813133964438748e-02 9.29218254085324879554264043690637e-01 1.53570007579775704570357319767027e-02
 	 4.33992891185142615206871141708689e-02 9.29218253512416048067734664073214e-01 2.73824573690697459227294530137442e-02 1.53570009782517478275076783233999e-02
 	 9.29218252541391342980148237984395e-01 2.73824594333478416041671721359307e-02 4.33992880252608154156845898796746e-02 1.53570016797273684733271537083965e-02
 	 2.90369914929698991201689750596415e-01 5.94849894428558720327337994149275e-02 6.50145095627445157582258161710342e-01 5.14405020894168202305607451307878e-02
 	 5.94849886320373416603857208428963e-02 6.50145097261727644699647044035373e-01 2.90369914106235027517755042936187e-01 5.14405007092475288765776042509970e-02
 	 6.50145090521004664907422920805402e-01 2.90369916954274109865963282572920e-01 5.94849925247212252266137966216775e-02 5.14405020890256678423035907599115e-02
 	 1.02579103608621838295000827656622e-01 1.45523272507306150602701677598816e-01 7.51897623884071997224509686930105e-01 5.51873925716194896340383024835319e-02
 	 1.45523268651140014195988214851241e-01 7.51897624119613472437606560561107e-01 1.02579107229246457855253993329825e-01 5.51873918420055370459564869634050e-02
 	 7.51897620213141881784224551665829e-01 1.02579105494409267285149667259248e-01 1.45523274292448850930625781074923e-01 5.51873931205727058513232918812719e-02
 	 8.91387185038082507926304742795764e-02 3.68934748340993656778863396539236e-01 5.41926533155198120184081744810101e-01 7.17179792716597097612662992105470e-02
 	 3.68934745933899244096920710944687e-01 5.41926532339293443030214803002309e-01 8.91387217268073683840157173108310e-02 7.17179807235874960946730993782694e-02
 	 5.41926530667821038100839814433130e-01 8.91387164194035602760024517010606e-02 3.68934752912775387745369926051353e-01 7.17179803611985439149023591198784e-02
 	 2.89306187176388229786994088499341e-01 2.27196143161295271983846078001079e-01 4.83497669662316498229159833499580e-01 9.50433968121971173292905632479233e-02
 	 2.27196140848565653058699354005512e-01 4.83497668078035791605628901379532e-01 2.89306191073398555335671744614956e-01 9.50433970883373530069704315792478e-02
 	 4.83497664803317683812622362893308e-01 2.89306187540847858929993208221276e-01 2.27196147655834457257384428885416e-01 9.50433977136816710018507592394599e-02
 	 ];
 
 	 case  11 

 	 % ALG. DEG.:   11
 	 % PTS CARD.:   27
 	 % NEG. W.  :    0
 	 % OUT PTS. :    0
 	 % M.E.INF. : 9.9e-16  

 	 xyw_bar=[
 	 3.17327240959096867900512961568893e-02 3.38153709534936494640611215345416e-02 9.34451904950596712318144909659168e-01 1.38072803213663795884880158837404e-02
 	 3.38153709571890753737832824299403e-02 9.34451904826319457164629511680687e-01 3.17327242164914258282237824460026e-02 1.38072803774423255585013237123349e-02
 	 9.34451904771643859781704577471828e-01 3.17327242005391030321703738081851e-02 3.38153710278170371861250487199868e-02 1.38072803979651364397751578394491e-02
 	 1.59837359530232481219513829273637e-01 2.47248304899685064639491827165330e-02 8.15437809979798977622067468473688e-01 2.17032351504748317883031205610678e-02
 	 2.47248303570562662223597527599850e-02 8.15437809424427229032517061568797e-01 1.59837360218516466581206714181462e-01 2.17032351438043447089398796379101e-02
 	 8.15437809808309377324064826098038e-01 1.59837360119939320135529214894632e-01 2.47248300717513025404059590073302e-02 2.17032349067440601120271281843088e-02
 	 2.95071648130748806493350855362223e-02 1.69403881037778814011574013420613e-01 8.01088954149146270644621381507022e-01 2.70341283615386021588200549103931e-02
 	 1.69403881176198312097369580442319e-01 8.01088954177139322965217616001610e-01 2.95071646466623649374128035560716e-02 2.70341282464158123854947035624718e-02
 	 8.01088953864914521396656255092239e-01 2.95071646435425549714892667907407e-02 1.69403881491542923631854478117020e-01 2.70341282912778362035322032852491e-02
 	 2.28225758126171518469504917447921e-02 3.82904117992176939999637852451997e-01 5.94273306195205908153411655803211e-01 2.73312406572563323936542190040200e-02
 	 3.82904118304647045345490141698974e-01 5.94273305888574410182911833544495e-01 2.28225758067784889604467934987042e-02 2.73312406542919779373068678296477e-02
 	 5.94273305139020657783532897155965e-01 2.28225758029828758310930680863748e-02 3.82904119057996483732608794525731e-01 2.73312407012505954873038405139596e-02
 	 3.60374165922343669965499657337205e-01 2.51549434748986507071055740425436e-02 6.14470890602757724430205144017236e-01 2.83362843929536971365745756656906e-02
 	 2.51549437595472435980070713412715e-02 6.14470889118425733776973629574059e-01 3.60374167122026967113868067826843e-01 2.83362847378892217242718487568709e-02
 	 6.14470889369991835060602625162574e-01 3.60374166818971664039139568558312e-01 2.51549438110365009002578062791144e-02 2.83362847709364637505125017469254e-02
 	 1.52649491517360930137314767307544e-01 1.30991018319333046848740309542336e-01 7.16359490163306023013944923150120e-01 4.85436675610489629106858444629324e-02
 	 1.30991018089064076290739535579633e-01 7.16359490961503864170367705810349e-01 1.52649490949432031783317142981105e-01 4.85436673950245395348090937659435e-02
 	 7.16359491749854693232180125050945e-01 1.52649491098747813744296308868797e-01 1.30991017151397493023523566080257e-01 4.85436673088348660809465684451425e-02
 	 3.48698928000996666298760828794912e-01 1.20982495295537784141259862735751e-01 5.30318576703465494048828077211510e-01 5.28872982756141626903279018279136e-02
 	 1.20982496547677217946237249179831e-01 5.30318577259008328006473220739281e-01 3.48698926193314440169501722266432e-01 5.28872983966971396685430306661146e-02
 	 5.30318576743719294341872227960266e-01 3.48698926655225394100057201285381e-01 1.20982496601055311558070570754353e-01 5.28872983994380027605863858752855e-02
 	 1.21263845339625003783012857638823e-01 3.19556007053755031233066574714030e-01 5.59180147606620048250647414533887e-01 5.51622534580109777357925793239701e-02
 	 3.19556006508081080763616910189739e-01 5.59180148330025272862542351504089e-01 1.21263845161893590862689507048344e-01 5.51622533214837437731681291097630e-02
 	 5.59180149334335019339903283253079e-01 1.21263845232059297440940781598329e-01 3.19556005433605683219155935148592e-01 5.51622532633941134450061838379042e-02
 	 2.79902470374426504129417025978910e-01 2.69674312574123831165451292690705e-01 4.50423217051449664705131681330386e-01 5.85279453929950163604267743266973e-02
 	 2.69674314114379864992798729872447e-01 4.50423216214933308521040089544840e-01 2.79902469670686881997312411840539e-01 5.85279453179242473148669034799241e-02
 	 4.50423217577915646625541512548807e-01 2.79902468937386739167294535945985e-01 2.69674313484697558696012720247381e-01 5.85279447979268480084513726069417e-02
 	 ];
 
 	 case  12 

 	 % ALG. DEG.:   12
 	 % PTS CARD.:   31
 	 % NEG. W.  :    0
 	 % OUT PTS. :    3
 	 % M.E.INF. : 2.2e-15  

 	 xyw_bar=[
 	 3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333425851918718763045e-01 2.14094186066215476371699111268754e-02
 	 3.01131981576056410865316337321929e-02 -9.47156971046334839048341791567509e-02 1.06460249894702796424894586380105e+00 8.36690970085460678255448452311782e-05
 	 -9.47156971046355516952175435108074e-02 1.06460249894703018469499511411414e+00 3.01131981576052698557077746954747e-02 8.36690970085403486590849841952888e-05
 	 1.06460249894702507766908183839405e+00 3.01131981576053600613285254894436e-02 -9.47156971046304307915164599762647e-02 8.36690970085543348671100472024875e-05
 	 3.28665292260699182236294291215017e-01 8.04564374091387701781741270679049e-03 6.63289063998386940745888296078192e-01 1.24136352531261286852304337458008e-02
 	 8.04564374091384232334789317064860e-03 6.63289063998386274612073520984268e-01 3.28665292260699848370109066308942e-01 1.24136352531261373588478136298363e-02
 	 6.63289063998386385634375983499922e-01 3.28665292260700070414713991340250e-01 8.04564374091354395091002515982836e-03 1.24136352531259690906706438795482e-02
 	 4.81249505497031090928139462903346e-02 9.23861895067511262880088906968012e-01 2.80131543827856210882032428344246e-02 1.62399519211943316188140329359157e-02
 	 9.23861895067510707768576594389742e-01 2.80131543827856974160361858139368e-02 4.81249505497035948153872197963210e-02 1.62399519211944461105634474051840e-02
 	 2.80131543827856280270971467416530e-02 4.81249505497032964429493517855008e-02 9.23861895067511040835483981936704e-01 1.62399519211943524354957446576009e-02
 	 1.42850847044125095486322152282810e-01 3.10192404956991518716336031502578e-02 8.26129912460175752642044244566932e-01 2.41940396511918405508811957815851e-02
 	 3.10192404956990790132476121243599e-02 8.26129912460175530597439319535624e-01 1.42850847044125428553229539829772e-01 2.41940396511918301425403399207426e-02
 	 8.26129912460175419575136857019970e-01 1.42850847044125428553229539829772e-01 3.10192404956991518716336031502578e-02 2.41940396511918440203281477351993e-02
 	 2.08191967793620313909030983268167e-01 7.64831204293204724997679022635566e-01 2.69768279131749055821387628384400e-02 2.69673648168111852740924661020472e-02
 	 7.64831204293203836819259322510334e-01 2.69768279131749472155021862818103e-02 2.08191967793621202087450683393399e-01 2.69673648168112060907741778237323e-02
 	 2.69768279131749680321838980034954e-02 2.08191967793620591464787139557302e-01 7.64831204293204391930771635088604e-01 2.69673648168111991518802739165039e-02
 	 4.25825852108942526363932756794384e-01 5.50395667341567640384880633064313e-01 2.37784805494898332511866101413034e-02 2.77744243468646991035431170757874e-02
 	 5.50395667341566752206460932939081e-01 2.37784805494897950872701386515473e-02 4.25825852108943470053503688177443e-01 2.77744243468646505312857897251888e-02
 	 2.37784805494898505984213699093743e-02 4.25825852108942748408537681825692e-01 5.50395667341567418340275708033005e-01 2.77744243468647060424370209830158e-02
 	 3.23868231681058882376333940555924e-01 7.98513424486459177931152453311370e-02 5.96280425870295283097277660999680e-01 4.70167780570433999076662701099849e-02
 	 7.98513424486458761597518218877667e-02 5.96280425870295172074975198484026e-01 3.23868231681058937887485171813751e-01 4.70167780570434068465601740172133e-02
 	 5.96280425870295394119580123515334e-01 3.23868231681059215443241328102886e-01 7.98513424486453904371785483817803e-02 4.70167780570434276632418857388984e-02
 	 1.54176409947630321806144593210774e-01 7.07647950434089900717538057506317e-01 1.38175639618279721965166118025081e-01 4.85934146427096300135950457388390e-02
 	 7.07647950434089789695235594990663e-01 1.38175639618279860743044196169649e-01 1.54176409947630349561720208839688e-01 4.85934146427096508302767574605241e-02
 	 1.38175639618279888498619811798562e-01 1.54176409947630516095173902613169e-01 7.07647950434089678672933132475009e-01 4.85934146427096438913828535532957e-02
 	 3.49383622277121108634645452184486e-01 5.24861097355139305520310699648689e-01 1.25755280367739530333892616908997e-01 5.85052798961143774580939691531967e-02
 	 5.24861097355139305520310699648689e-01 1.25755280367739502578317001280084e-01 3.49383622277121164145796683442313e-01 5.85052798961144121525634886893386e-02
 	 1.25755280367739613600619463795738e-01 3.49383622277121219656947914700140e-01 5.24861097355139083475705774617381e-01 5.85052798961143497025183535242832e-02
 	 2.99315867902454901816611254616873e-01 2.29249619433458784456547618901823e-01 4.71434512664086313726841126481304e-01 6.44083027823953296886472230653453e-02
 	 2.29249619433458812212123234530736e-01 4.71434512664086480260294820254785e-01 2.99315867902454735283157560843392e-01 6.44083027823953158108594152508886e-02
 	 4.71434512664086646793748514028266e-01 2.99315867902454846305460023359046e-01 2.29249619433458506900791462612688e-01 6.44083027823954268331618777665426e-02
 	 ];
 
 	 case  13 

 	 % ALG. DEG.:   13
 	 % PTS CARD.:   36
 	 % NEG. W.  :    0
 	 % OUT PTS. :    0
 	 % M.E.INF. : 3.1e-16  

 	 xyw_bar=[
 	 2.60167889975679161862487376311037e-02 2.40222268841206107459207430565584e-02 9.49960984118311535517875654477393e-01 8.07997928361406728137161792346888e-03
 	 2.40222266056275869650082910311539e-02 9.49960984299580535328288988239365e-01 2.60167890947918811761496726830956e-02 8.07997921304411462206651606265950e-03
 	 9.49960983828827321140408912469866e-01 2.60167891895750139075182971737377e-02 2.40222269815976649520727903563966e-02 8.07997937455707040954600017812481e-03
 	 1.88836306686942824450969879990225e-02 1.22080867325125158595611196687969e-01 8.59035502006180617939889998524450e-01 1.28920289088000090266250552417659e-02
 	 1.22080865615096983978205003040785e-01 8.59035503703662994823275766975712e-01 1.88836306812399934429436143545900e-02 1.28920287274982992703487028052223e-02
 	 8.59035502492874303293035609385697e-01 1.88836297321810057947555350210678e-02 1.22080867774944687442761903639621e-01 1.28920283815568172125498591640280e-02
 	 1.31639216755138704906258340088243e-01 2.28747090565116009686086329111276e-02 8.45486074188349756575178162165685e-01 1.65602436740405097037598380893542e-02
 	 2.28747093163145995275975508320698e-02 8.45486073780362001883759148768149e-01 1.31639216903323363894173780863639e-01 1.65602438534709420214774411306280e-02
 	 8.45486073324166143017066588072339e-01 1.31639217770324529199754692854185e-01 2.28747089055093277831787190734758e-02 1.65602437082687306557637896275992e-02
 	 2.07315314538747512418392915378718e-02 2.80737807897761981745787807085435e-01 6.98530660648363221909562525979709e-01 1.90231334641921927264274216895501e-02
 	 2.80737804477865227070765286043752e-01 6.98530664767438969064983211865183e-01 2.07315307546958038642515020910650e-02 1.90231326648886284180761663265002e-02
 	 6.98530660028288119534067845961545e-01 2.07315315728960856966889281238764e-02 2.80737808398815791299796273960965e-01 1.90231335679501740998365022505823e-02
 	 4.76642854053482611931258361437358e-01 5.04849500035025267052901654096786e-01 1.85076459114921210158399844658561e-02 1.93596509965653432172594250459952e-02
 	 5.04849495463474129408609769598115e-01 1.85076462972688683683575305849445e-02 4.76642858239257005692479651770554e-01 1.93596512890421047770050932967933e-02
 	 1.85076458585603301210120719133556e-02 4.76642857835611943873033169438713e-01 5.04849496305827649678121815668419e-01 1.93596508553124622975527557855457e-02
 	 3.02387014801126063900937879225239e-01 1.99757515090916715205793252607691e-02 6.77637233689782236822907179885078e-01 1.95837970771764770250289444675218e-02
 	 1.99757511746916083250624751599389e-02 6.77637233570464792009602206235286e-01 3.02387015254843571909759702975862e-01 1.95837968008982718504107367607503e-02
 	 6.77637231251634108986081628245302e-01 3.02387017597124940770925149990944e-01 1.99757511512409502429932217637543e-02 1.95837969292292982115277766297368e-02
 	 1.01539777415501644708406558947900e-01 1.19028238394272706246468374047254e-01 7.79431984190225635167337259190390e-01 3.02814962024566189380259118024696e-02
 	 1.19028239511687658480632023838552e-01 7.79431983178413778645676757150795e-01 1.01539777309898604507054642454023e-01 3.02814965520301696000338154135534e-02
 	 7.79431987279475047003529653011356e-01 1.01539774383865286089267954139359e-01 1.19028238336659666907202392849285e-01 3.02814958168418184303316564864872e-02
 	 2.49614419434176670886671445259708e-01 1.03589969002955084587114242822281e-01 6.46795611562868133503911849402357e-01 3.81640642640734065182961387563410e-02
 	 1.03589967812852373763021773811488e-01 6.46795611281056226360419714183081e-01 2.49614420906091427632134127634345e-01 3.81640640401437122863903539382591e-02
 	 6.46795616892277380749476378696272e-01 2.49614415590636157249448956463311e-01 1.03589967517086462001074664840417e-01 3.81640634828955618851331621499412e-02
 	 1.02029245899726006552299395480077e-01 2.76888289616485705479931311856490e-01 6.21082464483788232456618061405607e-01 3.81993383228338492796360981174075e-02
 	 2.76888294228863351964520234105294e-01 6.21082463846217014413753076951252e-01 1.02029241924919689132877920201281e-01 3.81993373874003269707877450400701e-02
 	 6.21082464001205591941356942697894e-01 1.02029246368156176205665985889937e-01 2.76888289630638217975189263597713e-01 3.81993384808104186056709750118898e-02
 	 2.26789276022037022961796992603922e-01 2.33728374217554740743807428771106e-01 5.39482349760408208538819962996058e-01 4.08471512984852971883675820663484e-02
 	 2.33728371859891315143187284775195e-01 5.39482360682531081330637334758649e-01 2.26789267457577659037326611723984e-01 4.08471498515373537818717863956408e-02
 	 5.39482355318465778282188693992794e-01 2.26789276022037022961796992603922e-01 2.33728368659497198756014313403284e-01 4.08471500540344423235694648610661e-02
 	 4.37409498131037799417697442549979e-01 9.59920858845616786769738837392651e-02 4.66598415984400549660904289339669e-01 4.10856921009676787082121052208095e-02
 	 9.59920836950179934277116444718558e-02 4.66598416159025475291599605043302e-01 4.37409500145956531280688750484842e-01 4.10856912417196587505863192291145e-02
 	 4.66598421189779177797873899180559e-01 4.37409494682125088438340299035190e-01 9.59920841280957892749370330420788e-02 4.10856915839857214645114424911299e-02
 	 3.80758577881083304372822340155835e-01 2.27855355201996984115808686510718e-01 3.91386066916919683755793357704533e-01 4.92567601152039125089743265561992e-02
 	 2.27855351201257216864348720264388e-01 3.91386077299870405177273369190516e-01 3.80758571498872377958377910545096e-01 4.92567607266731696813444330018683e-02
 	 3.91386069899954280870701950334478e-01 3.80758578955004756139146593341138e-01 2.27855351145040962990151456324384e-01 4.92567597078012661682144823771523e-02
 	 ];
 
 	 case  14 

 	 % ALG. DEG.:   14
 	 % PTS CARD.:   40
 	 % NEG. W.  :    0
 	 % OUT PTS. :    3
 	 % M.E.INF. : 9.5e-16  

 	 xyw_bar=[
 	 3.33333333333333092785011331216083e-01 3.33333333333333592385372412536526e-01 3.33333333333333370340767487505218e-01 1.72195485434007769520725616985146e-02
 	 1.22379568942328237346472974422795e-01 -2.21163705785205715048657992838343e-02 8.99736801636192295994476353371283e-01 1.22344300381037741735912316443091e-03
 	 -2.21163705785206478326987422633465e-02 8.99736801636192295994476353371283e-01 1.22379568942328265102048590051709e-01 1.22344300381036939426304677169810e-03
 	 8.99736801636192518039081278402591e-01 1.22379568942328126324170511907141e-01 -2.21163705785206443632517903097323e-02 1.22344300381037481527390919922027e-03
 	 2.56919955958364865100662655095221e-01 7.42961907725503301058722627203679e-01 1.18136316131889351765948958927765e-04 5.62580792932306904957640014686149e-03
 	 7.42961907725503079014117702172371e-01 1.18136316131866407337473734440181e-04 2.56919955958365031634116348868702e-01 5.62580792932310200932244370619628e-03
 	 1.18136316131851580872764995167046e-04 2.56919955958365364701023736415664e-01 7.42961907725502745947210314625409e-01 5.62580792932310114196070571779273e-03
 	 2.24127192652311194032144214816071e-02 2.12963622799149272790675269106941e-02 9.56290918454853922092695484025171e-01 6.18035569790843786197598319631652e-03
 	 2.12963622799148960540449593281664e-02 9.56290918454854033114997946540825e-01 2.24127192652310291975936706876382e-02 6.18035569790842658627338934707041e-03
 	 9.56290918454854033114997946540825e-01 2.24127192652310812392979499918511e-02 2.12963622799148856457041034673239e-02 6.18035569790842745363512733547395e-03
 	 1.07983678184864761995243043202208e-01 8.73605408045508569081505356734851e-01 1.84109137696266689232516000629403e-02 1.09577036773974862554625886446047e-02
 	 8.73605408045508458059202894219197e-01 1.84109137696266932093802637382396e-02 1.07983678184864845261969890088949e-01 1.09577036773974862554625886446047e-02
 	 1.84109137696267001482741676454680e-02 1.07983678184864984039847968233516e-01 8.73605408045508347036900431703543e-01 1.09577036773974845207391126677976e-02
 	 4.54399154241966474199188041893649e-01 8.40211993845903226674209918201086e-03 5.37198725819574396389555204223143e-01 1.14623422423098340200420253154334e-02
 	 8.40211993845906175704119078773147e-03 5.37198725819574396389555204223143e-01 4.54399154241966529710339273151476e-01 1.14623422423098392242124532458547e-02
 	 5.37198725819574507411857666738797e-01 4.54399154241966474199188041893649e-01 8.40211993845901838895429136755411e-03 1.14623422423098218769776934777838e-02
 	 2.60937045453201454936476011425839e-01 2.57352549212651951493491253586399e-02 7.13327699625533440119795614009490e-01 2.22957814896168095286643762165113e-02
 	 2.57352549212651743326674136369547e-02 7.13327699625533329097493151493836e-01 2.60937045453201510447627242683666e-01 2.22957814896168129981113281701255e-02
 	 7.13327699625533329097493151493836e-01 2.60937045453201454936476011425839e-01 2.57352549212652159660308370803250e-02 2.22957814896168060592174242628971e-02
 	 1.08651864883571383502136598053767e-01 4.73180071101256538579349353312864e-02 8.44030128006302948762140658800490e-01 2.33828660811444405398518142646935e-02
 	 4.73180071101256261023593197023729e-02 8.44030128006302948762140658800490e-01 1.08651864883571480646651252754964e-01 2.33828660811444231926170544966226e-02
 	 8.44030128006303059784443121316144e-01 1.08651864883571369624348790239310e-01 4.73180071101255705912080884445459e-02 2.33828660811444336009579103574652e-02
 	 3.63151187936801678191045539279003e-01 6.00340343076415372536303038941696e-01 3.65084689867829492726514217793010e-02 2.65580658068817858363797057563715e-02
 	 6.00340343076415261514000576426042e-01 3.65084689867830117226965569443564e-02 3.63151187936801733702196770536830e-01 2.65580658068818101225083694316709e-02
 	 3.65084689867830256004843647588132e-02 3.63151187936801733702196770536830e-01 6.00340343076415150491698113910388e-01 2.65580658068817858363797057563715e-02
 	 1.88070100346088509013497969135642e-01 7.41933483171285490520574512629537e-01 6.99964164826260004659275182348210e-02 3.08504847823172952125947432477915e-02
 	 7.41933483171285490520574512629537e-01 6.99964164826261114882299807504751e-02 1.88070100346088397991195506619988e-01 3.08504847823172952125947432477915e-02
 	 6.99964164826261114882299807504751e-02 1.88070100346088509013497969135642e-01 7.41933483171285379498272050113883e-01 3.08504847823173056209355991086341e-02
 	 3.93963054316368299634376626272569e-01 8.32192508939270103596186345384922e-02 5.22817694789704634494853507931111e-01 4.06810504210194656682375580203370e-02
 	 8.32192508939270658707698657963192e-02 5.22817694789704745517155970446765e-01 3.93963054316368133100922932499088e-01 4.06810504210194934238131736492505e-02
 	 5.22817694789704634494853507931111e-01 3.93963054316368355145527857530396e-01 8.32192508939270103596186345384922e-02 4.06810504210194726071314619275654e-02
 	 2.04286459600112130230442630818288e-01 1.37280352612603406825542151636910e-01 6.58433187787284435188439601915888e-01 4.36523943609726308712204456696782e-02
 	 1.37280352612603379069966536007996e-01 6.58433187787284657233044526947197e-01 2.04286459600111935941413321415894e-01 4.36523943609726239323265417624498e-02
 	 6.58433187787284546210742064431543e-01 2.04286459600112074719291399560461e-01 1.37280352612603379069966536007996e-01 4.36523943609726031156448300407646e-02
 	 3.15511927618559784303187143450486e-01 5.33376563749646703804785374813946e-01 1.51111508631793456380876250477741e-01 5.08937369794181942994626410836645e-02
 	 5.33376563749646703804785374813946e-01 1.51111508631793539647603097364481e-01 3.15511927618559728792035912192659e-01 5.08937369794181873605687371764361e-02
 	 1.51111508631793595158754328622308e-01 3.15511927618559839814338374708313e-01 5.33376563749646592782482912298292e-01 5.08937369794181942994626410836645e-02
 	 3.24023305595123767286480642724200e-01 2.28102761121843300440303892173688e-01 4.47873933283032876762064233844285e-01 5.38294513467465693401514670313190e-02
 	 2.28102761121843383707030739060428e-01 4.47873933283033043295517927617766e-01 3.24023305595123600753026948950719e-01 5.38294513467465138290002357734920e-02
 	 4.47873933283032876762064233844285e-01 3.24023305595123767286480642724200e-01 2.28102761121843411462606354689342e-01 5.38294513467466040346209865674609e-02
 	 ];
 
 	 case  15 

 	 % ALG. DEG.:   15
 	 % PTS CARD.:   46
 	 % NEG. W.  :    0
 	 % OUT PTS. :    0
 	 % M.E.INF. : 5.0e-16  

 	 xyw_bar=[
 	 3.33333292261711877646490620463737e-01 3.33333369225872422347833889944013e-01 3.33333338512415755516826720850077e-01 4.67496311751516707699849462187558e-02
 	 1.84348340534850607674854927608976e-02 2.10206116064859810377285498361744e-02 9.60544554340029055339300612104125e-01 5.02787179014453720210520160094347e-03
 	 2.10206198587913860309495106548638e-02 9.60544553925707256247790155612165e-01 1.84348262155014097629646130371839e-02 5.02787139275056790821416541348299e-03
 	 9.60544579872547044097075286117615e-01 1.84348154031861895252308869430635e-02 2.10206047242667663776938269393213e-02 5.02786478905690396162819766345820e-03
 	 9.44694948714600857275769385523745e-02 1.40084791735677512136692968169882e-02 8.91522025954972208161564140027622e-01 7.59867673634635927559788370899696e-03
 	 1.40084495619432081864319528108354e-02 8.91522111110737869310582937032450e-01 9.44694393273188826043451626901515e-02 7.59865799345664505448638692541863e-03
 	 8.91522077853913352818437942914898e-01 9.44693784006580272727404690158437e-02 1.40085437454286199088215880692587e-02 7.59869474612488368658880233397213e-03
 	 2.21059623335564947588238737807842e-01 1.46001805115354987840703415713506e-02 7.64340196152899498116539689362980e-01 1.07321324865132412818002904941750e-02
 	 1.46001627401248310761383564226890e-02 7.64340360199224910786597320111468e-01 2.21059477060650211299730472092051e-01 1.07321114512731463630501593797817e-02
 	 7.64340671414178274289952241815627e-01 2.21059339300117646542886973293207e-01 1.45999892857040791671607848911663e-02 1.07319767847375187319602929392204e-02
 	 1.83161349800822388644849780803270e-02 1.06831540445691930418803394786664e-01 8.74852324574225903575097618158907e-01 1.08308612234807866148145194529206e-02
 	 1.06831572649002698449649528811278e-01 8.74852289591818377267884443426738e-01 1.83161377591789520380416433908977e-02 1.08308658410427617602156402654145e-02
 	 8.74852363873429461627040382154519e-01 1.83161445101451825223648484097794e-02 1.06831491616425355850594769435702e-01 1.08308615252217244062649115221575e-02
 	 3.84111880036755948264470816866378e-01 1.35065559081281177933320236661530e-02 6.02381564055115892308833736024098e-01 1.22487188911526365120430170918553e-02
 	 1.35065505592318075428481805033698e-02 6.02381757760965697023891607386759e-01 3.84111691679802524923559303715592e-01 1.22487140790047254607086912869818e-02
 	 6.02381907561370333326067338930443e-01 3.84111551382989191427697051040013e-01 1.35065410556404752462356100295437e-02 1.22487092712827670692998793811057e-02
 	 1.76095784933983541542534823065580e-02 2.46869564037977468062834418560669e-01 7.35520857468624167374571243271930e-01 1.43978403598048129707898112883413e-02
 	 2.46869635891989791520728658724693e-01 7.35520785115458264336041338538053e-01 1.76095789925519996543812339950819e-02 1.43978444900901697556250269371958e-02
 	 7.35520958397237634152077134785941e-01 1.76095847380971885776101970577656e-02 2.46869456864665187678653524017136e-01 1.43978402635832804928872619143476e-02
 	 1.62110954488718303490024652546708e-02 4.20027449315563017329111517028650e-01 5.63761455235565200894143345067278e-01 1.49838004159798046655005521188286e-02
 	 4.20027557266491768928062811028212e-01 5.63761341880282595973028492153389e-01 1.62111008532256350989086968183983e-02 1.49838078776986880630284204585223e-02
 	 5.63761608884686626907978279632516e-01 1.62111046483847491550545072414025e-02 4.20027286466928606589732453358010e-01 1.49838046424340780304884646056962e-02
 	 9.45835540958728515636977363101323e-02 7.81937423335833381443649159336928e-02 8.27222703570543838047512963385088e-01 1.92326738724229000243592935248671e-02
 	 7.81936458310440452024181468004826e-02 8.27222782704063241965286579215899e-01 9.45835714648927128322952739836182e-02 1.92326699023918287756451661607571e-02
 	 8.27222507563532660768146342888940e-01 9.45835855871118430515664954327804e-02 7.81939068493554961802871616782795e-02 1.92326943965639568512315804582613e-02
 	 2.23020503669574404748132678832917e-01 6.90247082160969777087089482847659e-02 7.07954788114328659176521796325687e-01 2.08346679557464657672749552830282e-02
 	 6.90245901886988449946969126358454e-02 7.07954806600517905224023706978187e-01 2.23020603210783208147915956942597e-01 2.08346401681652748649753448262345e-02
 	 7.07955491881565479594939915841678e-01 2.23020657547390782626806071675674e-01 6.90238505710437377782540124826482e-02 2.08345114437783474059528998623136e-02
 	 9.13686563942972540530718106310815e-02 1.97441541806721687635217676870525e-01 7.11189801798981058311710512498394e-01 2.85769699461605707069278281551306e-02
 	 1.97441432426123636423298535191861e-01 7.11189907126281162241809852275765e-01 9.13686604475951735793159969034605e-02 2.85769693739790116293608690511974e-02
 	 7.11189569389398590359974150487687e-01 9.13686840082226642278939721109055e-02 1.97441746602378731534344069586950e-01 2.85769780896803320124899272514085e-02
 	 3.84530855617108480171140172387823e-01 7.30977274485456463004595661914209e-02 5.42371416934345873528400261420757e-01 2.90316671786509487851457578244663e-02
 	 7.30977106499786088988912524655461e-02 5.42371300752677920620214990776731e-01 3.84530988597343470480893756757723e-01 2.90316673383885573422880099769827e-02
 	 5.42371248933846916706613683345495e-01 3.84531035703955026416878126838128e-01 7.30977153621980568765081898163771e-02 2.90316846605046322005172498847969e-02
 	 2.15038812389831607285728409806325e-01 1.55893061854053271986231266055256e-01 6.29068125756115148483615939767333e-01 3.02402760842489307102631812540494e-02
 	 1.55892843158961874605594744025439e-01 6.29068363385964834932906342146453e-01 2.15038793455073262705923298199195e-01 3.02402589813331457158085413539084e-02
 	 6.29068887420656586151324063393986e-01 2.15038957479725206800580394883582e-01 1.55892155099618207048095541722432e-01 3.02404224262863285943581104220357e-02
 	 8.53189508964801446300185716609121e-02 3.59007499441052202460156195229501e-01 5.55673549662467625154249617480673e-01 3.25115238400138090169022575537383e-02
 	 3.59007386827242869209442233113805e-01 5.55673632624283819758659319631988e-01 8.53189805484733110318984472542070e-02 3.25115367139687458375441053703980e-02
 	 5.55673357289606983933083483861992e-01 8.53189724946177840969596672948683e-02 3.59007670215775231969956848843140e-01 3.25115194472903185074486032135610e-02
 	 2.04230402773116243020368187899294e-01 2.86157186802203589159887542336946e-01 5.09612410424680195575319885392673e-01 4.01871300956154198225078744144412e-02
 	 2.86156970552324307899283439837745e-01 5.09612567484152623187299013807205e-01 2.04230461963523013402266315097222e-01 4.01871451017934977478773816983448e-02
 	 5.09612717585778640660976179788122e-01 2.04230501683514309441136447276222e-01 2.86156780730707049897887372935656e-01 4.01872524603753555894058990816120e-02
 	 3.63891035835017306876437714890926e-01 1.81085979988352741010260160692269e-01 4.55022984176629896602150893158978e-01 4.13153506505143444660355100950255e-02
 	 1.81085896307873084332129565154901e-01 4.55023141283621535002623659238452e-01 3.63890962408505380665246775606647e-01 4.13153053080377585803972806388629e-02
 	 4.55022970575782392810282317441306e-01 3.63891085598291641467483259475557e-01 1.81085943825925910211083191825310e-01 4.13153263377575782899775447276625e-02
 	 ];
 
 	 case  16 

 	 % ALG. DEG.:   16
 	 % PTS CARD.:   51
 	 % NEG. W.  :    0
 	 % OUT PTS. :    6
 	 % M.E.INF. : 1.8e-15  

 	 xyw_bar=[
 	 2.60234718756303995412793028663145e-01 -7.79627663672445049558579199810993e-03 7.47561557880420446409175383450929e-01 2.65836591717333184728411232811141e-03
 	 -7.79627663672445830184143389374185e-03 7.47561557880420446409175383450929e-01 2.60234718756304106435095491178799e-01 2.65836591717332447470933942668125e-03
 	 7.47561557880420335386872920935275e-01 2.60234718756303995412793028663145e-01 -7.79627663672433079966594959842041e-03 2.65836591717333401568845729912027e-03
 	 8.76543641109690518931074620923027e-02 9.09443987549238164191933719848748e-01 2.90164833979278391495881805894896e-03 3.39960608267414953961860213382806e-03
 	 9.09443987549238164191933719848748e-01 2.90164833979279258857619794298444e-03 8.76543641109690380153196542778460e-02 3.39960608267415864691685101206531e-03
 	 2.90164833979275052153190550541240e-03 8.76543641109690102597440386489325e-02 9.09443987549238164191933719848748e-01 3.39960608267415300906555408744225e-03
 	 3.00672736234540771693701799449627e-01 7.00388362907983452743110319715925e-01 -1.06109914252416892566088790772483e-03 3.51912288753210766509282869662911e-03
 	 7.00388362907983674787715244747233e-01 -1.06109914252419733175780702794100e-03 3.00672736234540549649096874418319e-01 3.51912288753211894079542254587523e-03
 	 -1.06109914252421858212038774382791e-03 3.00672736234540771693701799449627e-01 7.00388362907983452743110319715925e-01 3.51912288753210896613543567923443e-03
 	 1.88499527230496406737803027908740e-02 1.64729254038669808979378217372869e-02 9.64677121873083298631001980538713e-01 4.04414427678121465165039793987489e-03
 	 1.64729254038669566118091580619875e-02 9.64677121873083409653304443054367e-01 1.88499527230496788376967742806301e-02 4.04414427678121291692692196306780e-03
 	 9.64677121873083409653304443054367e-01 1.88499527230495955709699273938895e-02 1.64729254038669947757256295517436e-02 4.04414427678121812109734989348908e-03
 	 9.84167329545414665270541831887385e-02 1.19409662687472318404191540253123e-02 8.89642300776711358878401370020583e-01 6.82944731326658605735646645484849e-03
 	 1.19409662687472370445895819557336e-02 8.89642300776711358878401370020583e-01 9.84167329545414526492663753742818e-02 6.82944731326657564901561059400592e-03
 	 8.89642300776711358878401370020583e-01 9.84167329545414526492663753742818e-02 1.19409662687471884723322546051349e-02 6.82944731326656437331301674475981e-03
 	 4.60403420093106141575134415688808e-01 1.17880011749511917629185120404145e-02 5.27808578731942645845265360549092e-01 1.04401945186626560896359094954278e-02
 	 1.17880011749511449253846606666229e-02 5.27808578731942534822962898033438e-01 4.60403420093106308108588109462289e-01 1.04401945186626560896359094954278e-02
 	 5.27808578731942423800660435517784e-01 4.60403420093106363619739340720116e-01 1.17880011749512125796002237620996e-02 1.04401945186626647632532893794632e-02
 	 7.38297774166404574547328820699477e-02 8.68681356668872584059215569141088e-01 5.74888659144869862416271644178778e-02 1.47306895843886381108900351932789e-02
 	 8.68681356668872473036913106625434e-01 5.74888659144870209360966839540197e-02 7.38297774166405129658841133277747e-02 1.47306895843886537234013189845427e-02
 	 5.74888659144869793027332605106494e-02 7.38297774166404852103084976988612e-02 8.68681356668872473036913106625434e-01 1.47306895843886398456135111700860e-02
 	 1.86109588791245589289857775838755e-01 7.87963168691664939125018918275600e-01 2.59272425170894438295476902567316e-02 1.49971864136080271917572659390316e-02
 	 7.87963168691665050147321380791254e-01 2.59272425170894195434190265814323e-02 1.86109588791245533778706544580928e-01 1.49971864136080098445225061709607e-02
 	 2.59272425170893917878434109525188e-02 1.86109588791245617045433391467668e-01 7.87963168691664939125018918275600e-01 1.49971864136080150486929341013820e-02
 	 3.83201939072719266476951816002838e-01 5.88557717697621440855471064423909e-01 2.82403432296592926675771195732523e-02 1.65631870790469599985517135110058e-02
 	 5.88557717697621218810866139392601e-01 2.82403432296592579731076000371104e-02 3.83201939072719544032707972291973e-01 1.65631870790469461207639056965490e-02
 	 2.82403432296592718508954078515671e-02 3.83201939072719321988103047260665e-01 5.88557717697621329833168601908255e-01 1.65631870790469495902108576501632e-02
 	 1.86107863137863926983328610731405e-01 4.21292162463990718657136369529326e-02 7.71762920615737035845427271851804e-01 1.97481718009701670557554109564080e-02
 	 4.21292162463990441101380213240191e-02 7.71762920615737035845427271851804e-01 1.86107863137863871472177379473578e-01 1.97481718009701739946493148636364e-02
 	 7.71762920615737035845427271851804e-01 1.86107863137863926983328610731405e-01 4.21292162463990371712441174167907e-02 1.97481718009701670557554109564080e-02
 	 3.29435605315130441983484388401848e-01 3.83979662592876733184787951813632e-02 6.32166428425581905514718528138474e-01 2.12031456417453544049944724747547e-02
 	 3.83979662592876594406909873669065e-02 6.32166428425581905514718528138474e-01 3.29435605315130386472333157144021e-01 2.12031456417453613438883763819831e-02
 	 6.32166428425581794492416065622820e-01 3.29435605315130497494635619659675e-01 3.83979662592877080129483147175051e-02 2.12031456417453509355475205211405e-02
 	 2.72057145537797639978094821344712e-01 6.51051865673838658921113164979033e-01 7.68909887883637566119432449340820e-02 2.44486585253550423035839855856466e-02
 	 6.51051865673838547898810702463379e-01 7.68909887883636317118529746039712e-02 2.72057145537797806511548515118193e-01 2.44486585253550561813717934001033e-02
 	 7.68909887883636733452163980473415e-02 2.72057145537797639978094821344712e-01 6.51051865673838658921113164979033e-01 2.44486585253550284257961777711898e-02
 	 1.41235187360025743474878368033387e-01 7.29399431838533618410735925863264e-01 1.29365380801440665869961321732262e-01 2.81498869289737788101923854355846e-02
 	 7.29399431838533618410735925863264e-01 1.29365380801440610358810090474435e-01 1.41235187360025771230453983662301e-01 2.81498869289737961574271452036555e-02
 	 1.29365380801440610358810090474435e-01 1.41235187360025798986029599291214e-01 7.29399431838533618410735925863264e-01 2.81498869289737961574271452036555e-02
 	 4.49015766811866112728779398821644e-01 9.76217927915696837004588815034367e-02 4.53362440396564148059610488417093e-01 3.69021345675010631337542577057320e-02
 	 9.76217927915697392116101127612637e-02 4.53362440396564259081912950932747e-01 4.49015766811866057217628167563817e-01 3.69021345675010700726481616129604e-02
 	 4.53362440396564259081912950932747e-01 4.49015766811866057217628167563817e-01 9.76217927915696837004588815034367e-02 3.69021345675010700726481616129604e-02
 	 2.66798474540568653079475325284875e-01 1.37399173718296491175649975957640e-01 5.95802351741134827989299083128572e-01 3.70477486131056410467721207169234e-02
 	 1.37399173718296463420074360328726e-01 5.95802351741134939011601545644226e-01 2.66798474540568597568324094027048e-01 3.70477486131056341078782168096950e-02
 	 5.95802351741134939011601545644226e-01 2.66798474540568653079475325284875e-01 1.37399173718296407908923129070899e-01 3.70477486131056410467721207169234e-02
 	 2.82382350376655322232721800901345e-01 5.34917405870669493417324247275246e-01 1.82700243752675128838802720565582e-01 4.13325500853524083200163374840486e-02
 	 5.34917405870669382395021784759592e-01 1.82700243752675295372256414339063e-01 2.82382350376655322232721800901345e-01 4.13325500853523944422285296695918e-02
 	 1.82700243752675239861105183081236e-01 2.82382350376655322232721800901345e-01 5.34917405870669382395021784759592e-01 4.13325500853524083200163374840486e-02
 	 3.49747383256928001671326455834787e-01 2.38132117614774041491898515232606e-01 4.12120499128297956836775028932607e-01 4.73190930971962089857107969237404e-02
 	 2.38132117614773985980747283974779e-01 4.12120499128298012347926260190434e-01 3.49747383256928057182477687092614e-01 4.73190930971962159246047008309688e-02
 	 4.12120499128298067859077491448261e-01 3.49747383256928001671326455834787e-01 2.38132117614773930469596052716952e-01 4.73190930971961881690290852020553e-02
 	 ];
 
 	 case  17 

 	 % ALG. DEG.:   17
 	 % PTS CARD.:   57
 	 % NEG. W.  :    0
 	 % OUT PTS. :    0
 	 % M.E.INF. : 5.8e-16  

 	 xyw_bar=[
 	 1.47209661956265915744301864265253e-02 1.76399848555067323419009994722728e-02 9.67639048948866653532263626402710e-01 3.37703600809952965422500170689091e-03
 	 1.76399848555067323419009994722728e-02 9.67639048948866653532263626402710e-01 1.47209661956265724924719506816473e-02 3.37703600809952965422500170689091e-03
 	 9.67639048948866653532263626402710e-01 1.47209661956265915744301864265253e-02 1.76399848555067531585827111939579e-02 3.37703600809952965422500170689091e-03
 	 7.58278723376997892957618319087487e-02 7.69747584377225568808000844001072e-03 9.16474651818527941138370351836784e-01 3.81992567019393035668284319683607e-03
 	 7.69747584377225568808000844001072e-03 9.16474651818527941138370351836784e-01 7.58278723376998309291252553521190e-02 3.81992567019393035668284319683607e-03
 	 9.16474651818527941138370351836784e-01 7.58278723376997892957618319087487e-02 7.69747584377226956586781625446747e-03 3.81992567019393035668284319683607e-03
 	 3.10585516558464003189499180734856e-01 8.32009195104604741544118695628640e-03 6.81094391490490003171487387589877e-01 6.62402608144134919659906657329884e-03
 	 8.32009195104604741544118695628640e-03 6.81094391490490003171487387589877e-01 3.10585516558464003189499180734856e-01 6.62402608144134919659906657329884e-03
 	 6.81094391490490003171487387589877e-01 3.10585516558464003189499180734856e-01 8.32009195104599363901343167526647e-03 6.62402608144134919659906657329884e-03
 	 1.44883146249404001770777483670827e-02 9.01640395800713256857505939478870e-02 8.95347645794988267198277753777802e-01 7.28490073487348104713801077991775e-03
 	 9.01640395800713256857505939478870e-02 8.95347645794988267198277753777802e-01 1.44883146249403793603960366453975e-02 7.28490073487348104713801077991775e-03
 	 8.95347645794988267198277753777802e-01 1.44883146249404001770777483670827e-02 9.01640395800713256857505939478870e-02 7.28490073487348104713801077991775e-03
 	 1.78143852253019957521118499244039e-01 1.64238580465156137899729316131925e-02 8.05432289700464432158355521096382e-01 9.96500247340540747098547313953532e-03
 	 1.64238580465156137899729316131925e-02 8.05432289700464432158355521096382e-01 1.78143852253019985276694114872953e-01 9.96500247340540747098547313953532e-03
 	 8.05432289700464432158355521096382e-01 1.78143852253019957521118499244039e-01 1.64238580465156103205259796595783e-02 9.96500247340540747098547313953532e-03
 	 4.66133550795934714638946161358035e-01 1.23355817687377852104324560400528e-02 5.21530867435327394332489348016679e-01 1.02763450985959500783017972480593e-02
 	 1.23355817687377852104324560400528e-02 5.21530867435327394332489348016679e-01 4.66133550795934770150097392615862e-01 1.02763450985959500783017972480593e-02
 	 5.21530867435327394332489348016679e-01 4.66133550795934714638946161358035e-01 1.23355817687378910285644906252855e-02 1.02763450985959500783017972480593e-02
 	 1.47365661662822756972079929482788e-02 2.09325171157457634585341565980343e-01 7.75938262676260093186897393024992e-01 1.03094913037771809816156576289359e-02
 	 2.09325171157457634585341565980343e-01 7.75938262676260093186897393024992e-01 1.47365661662822722277610409946647e-02 1.03094913037771809816156576289359e-02
 	 7.75938262676260093186897393024992e-01 1.47365661662822756972079929482788e-02 2.09325171157457634585341565980343e-01 1.03094913037771809816156576289359e-02
 	 1.43325355478803179165581127563200e-02 3.58602315795423676192399398132693e-01 6.27065148656696047524405912554357e-01 1.16022291368002225531164128824457e-02
 	 3.58602315795423676192399398132693e-01 6.27065148656696047524405912554357e-01 1.43325355478802762831946893129498e-02 1.16022291368002225531164128824457e-02
 	 6.27065148656696047524405912554357e-01 1.43325355478803179165581127563200e-02 3.58602315795423620681248166874866e-01 1.16022291368002225531164128824457e-02
 	 7.53033493906421630414982359980058e-02 5.59681329947420772019661683316372e-02 8.68728517614615669550914844876388e-01 1.24697437874820918346507880869467e-02
 	 5.59681329947420772019661683316372e-02 8.68728517614615669550914844876388e-01 7.53033493906422046748616594413761e-02 1.24697437874820918346507880869467e-02
 	 8.68728517614615669550914844876388e-01 7.53033493906421630414982359980058e-02 5.59681329947421674075869191256061e-02 1.24697437874820918346507880869467e-02
 	 3.17339393501727851099047938987496e-01 5.16948784187994947458832939446438e-02 6.30965728079472598643917535810033e-01 1.96324795155319517525605022001400e-02
 	 5.16948784187994947458832939446438e-02 6.30965728079472598643917535810033e-01 3.17339393501727906610199170245323e-01 1.96324795155319517525605022001400e-02
 	 6.30965728079472598643917535810033e-01 3.17339393501727851099047938987496e-01 5.16948784187995502570345252024708e-02 1.96324795155319517525605022001400e-02
 	 7.53542654903545661149166789982701e-02 1.54598211665730095321080739267927e-01 7.70047522843915421830729428620543e-01 2.01015070751451897790307299374035e-02
 	 1.54598211665730095321080739267927e-01 7.70047522843915421830729428620543e-01 7.53542654903545106037654477404431e-02 2.01015070751451897790307299374035e-02
 	 7.70047522843915421830729428620543e-01 7.53542654903545661149166789982701e-02 1.54598211665730012054353892381187e-01 2.01015070751451897790307299374035e-02
 	 1.82570774440876770761121861141874e-01 2.66530059635146754359880105766933e-01 5.50899165923976474878998033091193e-01 2.09060455685651375157263487380987e-02
 	 2.66530059635146754359880105766933e-01 5.50899165923976474878998033091193e-01 1.82570774440876770761121861141874e-01 2.09060455685651375157263487380987e-02
 	 5.50899165923976474878998033091193e-01 1.82570774440876770761121861141874e-01 2.66530059635146754359880105766933e-01 2.09060455685651375157263487380987e-02
 	 1.79710032722074752875585090805544e-01 8.15862277549058850789620578325412e-02 7.38703739523019375923240659176372e-01 2.13322476716037226496247569684783e-02
 	 8.15862277549058850789620578325412e-02 7.38703739523019375923240659176372e-01 1.79710032722074752875585090805544e-01 2.13322476716037226496247569684783e-02
 	 7.38703739523019375923240659176372e-01 1.79710032722074752875585090805544e-01 8.15862277549058712011742500180844e-02 2.13322476716037226496247569684783e-02
 	 1.75807941715677179539412122721842e-01 1.80353640520785951650850620353594e-01 6.43838417763536896565312872553477e-01 2.37493771891846804089620803779326e-02
 	 1.80353640520785951650850620353594e-01 6.43838417763536896565312872553477e-01 1.75807941715677151783836507092929e-01 2.37493771891846804089620803779326e-02
 	 6.43838417763536896565312872553477e-01 1.75807941715677179539412122721842e-01 1.80353640520785923895275004724681e-01 2.37493771891846804089620803779326e-02
 	 4.58731499688783517232337771929451e-01 4.74126007719326725275976741613704e-01 6.71424925918898130028367177146720e-02 2.47770240081320279601495570886982e-02
 	 4.74126007719326725275976741613704e-01 6.71424925918898130028367177146720e-02 4.58731499688783406210035309413797e-01 2.47770240081320279601495570886982e-02
 	 6.71424925918898130028367177146720e-02 4.58731499688783517232337771929451e-01 4.74126007719326725275976741613704e-01 2.47770240081320279601495570886982e-02
 	 7.53196920743743475457065983391658e-02 2.93754105362967754544456511212047e-01 6.30926202562657856276473467005417e-01 2.55855702422726240252703888700125e-02
 	 2.93754105362967754544456511212047e-01 6.30926202562657856276473467005417e-01 7.53196920743743891790700217825361e-02 2.55855702422726240252703888700125e-02
 	 6.30926202562657856276473467005417e-01 7.53196920743743475457065983391658e-02 2.93754105362967810055607742469874e-01 2.55855702422726240252703888700125e-02
 	 1.63305177925384792381180432130350e-01 3.84251187765941693008642232598504e-01 4.52443634308673514610177335271146e-01 3.13108588033011586704290607485746e-02
 	 3.84251187765941693008642232598504e-01 4.52443634308673514610177335271146e-01 1.63305177925384792381180432130350e-01 3.13108588033011586704290607485746e-02
 	 4.52443634308673514610177335271146e-01 1.63305177925384792381180432130350e-01 3.84251187765941693008642232598504e-01 3.13108588033011586704290607485746e-02
 	 3.11813568699614629409211374877486e-01 1.39501525996608172208723885887593e-01 5.48684905303777226137640354863834e-01 3.15903625494375200899455080616463e-02
 	 1.39501525996608172208723885887593e-01 5.48684905303777226137640354863834e-01 3.11813568699614629409211374877486e-01 3.15903625494375200899455080616463e-02
 	 5.48684905303777226137640354863834e-01 3.11813568699614629409211374877486e-01 1.39501525996608144453148270258680e-01 3.15903625494375200899455080616463e-02
 	 3.06048259112027931649890888365917e-01 2.67720653545962083974529832630651e-01 4.26231087342009984375579279003432e-01 3.86191604154902459744569398480962e-02
 	 2.67720653545962083974529832630651e-01 4.26231087342009984375579279003432e-01 3.06048259112027931649890888365917e-01 3.86191604154902459744569398480962e-02
 	 4.26231087342009984375579279003432e-01 3.06048259112027931649890888365917e-01 2.67720653545962083974529832630651e-01 3.86191604154902459744569398480962e-02
 	 ];
 
 	 case  18 

 	 % ALG. DEG.:   18
 	 % PTS CARD.:   64
 	 % NEG. W.  :    1
 	 % OUT PTS. :    0
 	 % M.E.INF. : 4.9e-16  

 	 xyw_bar=[
 	 3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333425851918718763045e-01 -1.25845221584873240283641848691332e-01
 	 9.22437177016372997218063289892598e-03 9.54037638902169304877531885722419e-03 9.81235251840814504120658057217952e-01 1.29727110524691319551338075655167e-03
 	 9.54037638902169304877531885722419e-03 9.81235251840814504120658057217952e-01 9.22437177016377507499100829591043e-03 1.29727110524691319551338075655167e-03
 	 9.81235251840814504120658057217952e-01 9.22437177016372997218063289892598e-03 9.54037638902176590716130988312216e-03 1.29727110524691319551338075655167e-03
 	 1.15881748682225369645459878142901e-02 5.99102344454169738474647033399378e-02 9.28501590686360511739394496544264e-01 4.40434451590567769413420151636274e-03
 	 5.99102344454169738474647033399378e-02 9.28501590686360511739394496544264e-01 1.15881748682225005353529923013411e-02 4.40434451590567769413420151636274e-03
 	 9.28501590686360511739394496544264e-01 1.15881748682225369645459878142901e-02 5.99102344454169530307829916182527e-02 4.40434451590567769413420151636274e-03
 	 1.37109449232486801939501219749218e-01 8.13423222383992221284909618361780e-03 8.54756318543673287990714015904814e-01 4.55380277877178666484647706624855e-03
 	 8.13423222383992221284909618361780e-03 8.54756318543673287990714015904814e-01 1.37109449232486801939501219749218e-01 4.55380277877178666484647706624855e-03
 	 8.54756318543673287990714015904814e-01 1.37109449232486801939501219749218e-01 8.13423222383991006978476434596814e-03 4.55380277877178666484647706624855e-03
 	 5.40746581302322046336250593867589e-02 1.59640348658512051804869713578228e-02 9.29961307003916659574827008327702e-01 5.25221168748300770712233997983276e-03
 	 1.59640348658512051804869713578228e-02 9.29961307003916659574827008327702e-01 5.40746581302321560613677320361603e-02 5.25221168748300770712233997983276e-03
 	 9.29961307003916659574827008327702e-01 5.40746581302322046336250593867589e-02 1.59640348658511357915479322855390e-02 5.25221168748300770712233997983276e-03
 	 1.23562085317423936414726881594106e-02 1.56321458214785352636155835170939e-01 8.31322333253472267600159284484107e-01 7.37463787220985107773163491629020e-03
 	 1.56321458214785352636155835170939e-01 8.31322333253472267600159284484107e-01 1.23562085317423520081092647160403e-02 7.37463787220985107773163491629020e-03
 	 8.31322333253472267600159284484107e-01 1.23562085317423936414726881594106e-02 1.56321458214785324880580219542026e-01 7.37463787220985107773163491629020e-03
 	 2.54443010786652157317888622856117e-01 1.32280360805979747829130843683743e-02 7.32328953132749815857494013471296e-01 8.83478687259323623715001616574227e-03
 	 1.32280360805979747829130843683743e-02 7.32328953132749815857494013471296e-01 2.54443010786652212829039854113944e-01 8.83478687259323623715001616574227e-03
 	 7.32328953132749815857494013471296e-01 2.54443010786652157317888622856117e-01 1.32280360805980268246173636725871e-02 8.83478687259323623715001616574227e-03
 	 1.22006545481682538573098284473417e-02 4.38518077301824849012490403765696e-01 5.49281268150006996009437898464967e-01 9.64777642488273169674162232922754e-03
 	 4.38518077301824849012490403765696e-01 5.49281268150006996009437898464967e-01 1.22006545481680994669204665115103e-02 9.64777642488273169674162232922754e-03
 	 5.49281268150006996009437898464967e-01 1.22006545481682538573098284473417e-02 4.38518077301824737990187941250042e-01 9.64777642488273169674162232922754e-03
 	 3.95592717135403115769065607310040e-01 1.29052660579212063324261450247832e-02 5.91502016806675690041572579502827e-01 9.86709978042134457087986021406323e-03
 	 1.29052660579212063324261450247832e-02 5.91502016806675690041572579502827e-01 3.95592717135403115769065607310040e-01 9.86709978042134457087986021406323e-03
 	 5.91502016806675690041572579502827e-01 3.95592717135403115769065607310040e-01 1.29052660579211941893618131871335e-02 9.86709978042134457087986021406323e-03
 	 1.39894903845946654008525200652002e-02 2.87578430196016754383236957437475e-01 6.98432079419388562868675762729254e-01 1.02607131892317442883877376402779e-02
 	 2.87578430196016754383236957437475e-01 6.98432079419388562868675762729254e-01 1.39894903845947382592385110910982e-02 1.02607131892317442883877376402779e-02
 	 6.98432079419388562868675762729254e-01 1.39894903845946654008525200652002e-02 2.87578430196016754383236957437475e-01 1.02607131892317442883877376402779e-02
 	 6.12426222536310363442346726969845e-02 8.21310314077598468962548849958694e-02 8.56626346338609012676101883698720e-01 1.17966959830773238104839606421592e-02
 	 8.21310314077598468962548849958694e-02 8.56626346338609012676101883698720e-01 6.12426222536310849164920000475831e-02 1.17966959830773238104839606421592e-02
 	 8.56626346338609012676101883698720e-01 6.12426222536310363442346726969845e-02 8.21310314077599579185573475115234e-02 1.17966959830773238104839606421592e-02
 	 1.40187294281714058818977264309069e-01 5.09884646182077044818115041380224e-02 8.08824241100078222821423423738452e-01 1.29855075351113964654548027510828e-02
 	 5.09884646182077044818115041380224e-02 8.08824241100078222821423423738452e-01 1.40187294281714058818977264309069e-01 1.29855075351113964654548027510828e-02
 	 8.08824241100078222821423423738452e-01 1.40187294281714058818977264309069e-01 5.09884646182077183595993119524792e-02 1.29855075351113964654548027510828e-02
 	 6.59534606576724913296061458822805e-02 1.82306457551239170511436782362580e-01 7.51740081791088310403381456126226e-01 1.63182058245727769352217961795759e-02
 	 1.82306457551239170511436782362580e-01 7.51740081791088310403381456126226e-01 6.59534606576724913296061458822805e-02 1.63182058245727769352217961795759e-02
 	 7.51740081791088310403381456126226e-01 6.59534606576724913296061458822805e-02 1.82306457551239198267012397991493e-01 1.63182058245727769352217961795759e-02
 	 6.24139442088075535597369025708758e-02 4.11715469349845597513137818168616e-01 5.25870586441346876682700894889422e-01 1.75256165062230781503949117450247e-02
 	 4.11715469349845597513137818168616e-01 5.25870586441346876682700894889422e-01 6.24139442088075258041612869419623e-02 1.75256165062230781503949117450247e-02
 	 5.25870586441346876682700894889422e-01 6.24139442088075535597369025708758e-02 4.11715469349845597513137818168616e-01 1.75256165062230781503949117450247e-02
 	 7.43017152789951784708222248809761e-02 2.93703944612865197605344746989431e-01 6.31994340108139596168257412500679e-01 1.84118314428711564445517723243029e-02
 	 2.93703944612865197605344746989431e-01 6.31994340108139596168257412500679e-01 7.43017152789952062263978405098896e-02 1.84118314428711564445517723243029e-02
 	 6.31994340108139596168257412500679e-01 7.43017152789951784708222248809761e-02 2.93703944612865197605344746989431e-01 1.84118314428711564445517723243029e-02
 	 2.55758429076780702260407451831270e-01 6.81831468635734766481348856359546e-02 6.76058424059645779458094239089405e-01 1.90689705940650690296767066911343e-02
 	 6.81831468635734766481348856359546e-02 6.76058424059645779458094239089405e-01 2.55758429076780702260407451831270e-01 1.90689705940650690296767066911343e-02
 	 6.76058424059645779458094239089405e-01 2.55758429076780702260407451831270e-01 6.81831468635735182814983090793248e-02 1.90689705940650690296767066911343e-02
 	 3.89748898002750199243138240490225e-01 6.69978935694291871660865922422090e-02 5.43253208427820655224138590710936e-01 2.05821846859529436479796515868657e-02
 	 6.69978935694291871660865922422090e-02 5.43253208427820655224138590710936e-01 3.89748898002750143731987009232398e-01 2.05821846859529436479796515868657e-02
 	 5.43253208427820655224138590710936e-01 3.89748898002750199243138240490225e-01 6.69978935694291455327231687988387e-02 2.05821846859529436479796515868657e-02
 	 1.55168432990029553364408343441028e-01 1.40086682350669539109944139454456e-01 7.04744884659300852014496285846690e-01 2.35891412493054869470299195199914e-02
 	 1.40086682350669539109944139454456e-01 7.04744884659300852014496285846690e-01 1.55168432990029581119983959069941e-01 2.35891412493054869470299195199914e-02
 	 7.04744884659300852014496285846690e-01 1.55168432990029553364408343441028e-01 1.40086682350669594621095370712283e-01 2.35891412493054869470299195199914e-02
 	 1.70472927860750189665850484743714e-01 2.61254002060855416988971455793944e-01 5.68273070078394448856329290720169e-01 3.10909522631880735898945999906573e-02
 	 2.61254002060855416988971455793944e-01 5.68273070078394448856329290720169e-01 1.70472927860750189665850484743714e-01 3.10909522631880735898945999906573e-02
 	 5.68273070078394448856329290720169e-01 1.70472927860750189665850484743714e-01 2.61254002060855361477820224536117e-01 3.10909522631880735898945999906573e-02
 	 2.86469196549182070210548545219353e-01 1.63053894264229143340472205636615e-01 5.50476909186588758693403633515118e-01 3.20399669872254433400371453899425e-02
 	 1.63053894264229143340472205636615e-01 5.50476909186588758693403633515118e-01 2.86469196549182125721699776477180e-01 3.20399669872254433400371453899425e-02
 	 5.50476909186588758693403633515118e-01 2.86469196549182070210548545219353e-01 1.63053894264229171096047821265529e-01 3.20399669872254433400371453899425e-02
 	 1.55079032148931039714057078526821e-01 4.08496920155350773384839158097748e-01 4.36424047695718186901103763375431e-01 3.26604177729548361686084945176844e-02
 	 4.08496920155350773384839158097748e-01 4.36424047695718186901103763375431e-01 1.55079032148931039714057078526821e-01 3.26604177729548361686084945176844e-02
 	 4.36424047695718186901103763375431e-01 1.55079032148931039714057078526821e-01 4.08496920155350773384839158097748e-01 3.26604177729548361686084945176844e-02
 	 3.06409473651319375075985362855135e-01 3.04333597697522717506757317096344e-01 3.89256928651157851906106088790693e-01 7.77196054569972011316636439914873e-02
 	 3.04333597697522717506757317096344e-01 3.89256928651157851906106088790693e-01 3.06409473651319375075985362855135e-01 7.77196054569972011316636439914873e-02
 	 3.89256928651157851906106088790693e-01 3.06409473651319375075985362855135e-01 3.04333597697522828529059779611998e-01 7.77196054569972011316636439914873e-02
 	 ];
 
 	 case  19 

 	 % ALG. DEG.:   19
 	 % PTS CARD.:   70
 	 % NEG. W.  :    0
 	 % OUT PTS. :    0
 	 % M.E.INF. : 4.2e-16  

 	 xyw_bar=[
 	 3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333425851918718763045e-01 3.04629617601474174393150207151848e-02
 	 1.03766565457017543444573348665472e-02 1.44810472658998996842161233189472e-02 9.75142296188398316481027450208785e-01 1.95018195806569887354453562977596e-03
 	 1.44810472658998996842161233189472e-02 9.75142296188398316481027450208785e-01 1.03766565457017456708399549825117e-02 1.95018195806569887354453562977596e-03
 	 9.75142296188398316481027450208785e-01 1.03766565457017543444573348665472e-02 1.44810472658999291745152149246678e-02 1.95018195806569887354453562977596e-03
 	 5.28956402129591926608576102353254e-02 8.92711734256453406932685368246894e-03 9.38177242444476200411429545056308e-01 2.76194461717317509691360655210701e-03
 	 8.92711734256453406932685368246894e-03 9.38177242444476200411429545056308e-01 5.28956402129592273553271297714673e-02 2.76194461717317509691360655210701e-03
 	 9.38177242444476200411429545056308e-01 5.28956402129591926608576102353254e-02 8.92711734256460692771284470836690e-03 2.76194461717317509691360655210701e-03
 	 1.30940571465279348428722983044281e-01 6.39606356170804597055523643689412e-03 8.62663364973012547487485335295787e-01 3.53317943173294237227399783307646e-03
 	 6.39606356170804597055523643689412e-03 8.62663364973012547487485335295787e-01 1.30940571465279376184298598673195e-01 3.53317943173294237227399783307646e-03
 	 8.62663364973012547487485335295787e-01 1.30940571465279348428722983044281e-01 6.39606356170810408379168165993178e-03 3.53317943173294237227399783307646e-03
 	 1.26681865510903015475951960411294e-02 7.29082350599041251193455082102446e-02 9.14423578389005564659441915864591e-01 5.15202907769013885436182320631815e-03
 	 7.29082350599041251193455082102446e-02 9.14423578389005564659441915864591e-01 1.26681865510902547100613446673378e-02 5.15202907769013885436182320631815e-03
 	 9.14423578389005564659441915864591e-01 1.26681865510903015475951960411294e-02 7.29082350599041389971333160247013e-02 5.15202907769013885436182320631815e-03
 	 2.46613960104767354630794784497994e-01 1.13210284482716614512520791890893e-02 7.42065011446961020347146131825866e-01 7.31171016627054411868291694531763e-03
 	 1.13210284482716614512520791890893e-02 7.42065011446961020347146131825866e-01 2.46613960104767326875219168869080e-01 7.31171016627054411868291694531763e-03
 	 7.42065011446961020347146131825866e-01 2.46613960104767354630794784497994e-01 1.13210284482716250220590836761403e-02 7.31171016627054411868291694531763e-03
 	 1.24330000962839457012698574089882e-02 1.71507203857275219993638870619179e-01 8.16059796046440855121772983693518e-01 7.38947305176194165626801790835998e-03
 	 1.71507203857275219993638870619179e-01 8.16059796046440855121772983693518e-01 1.24330000962838971290125300583895e-02 7.38947305176194165626801790835998e-03
 	 8.16059796046440855121772983693518e-01 1.24330000962839457012698574089882e-02 1.71507203857275192238063254990266e-01 7.38947305176194165626801790835998e-03
 	 9.45870867926038548512135406554080e-03 4.54553083656948797575125809089513e-01 5.35988207663790849899498880404280e-01 7.54778605267613766222067894773318e-03
 	 4.54553083656948797575125809089513e-01 5.35988207663790849899498880404280e-01 9.45870867926035252537531050620601e-03 7.54778605267613766222067894773318e-03
 	 5.35988207663790849899498880404280e-01 9.45870867926038548512135406554080e-03 4.54553083656948742063974577831686e-01 7.54778605267613766222067894773318e-03
 	 3.82677127432900721082376094273059e-01 1.03041271596045185682255507231275e-02 6.07018745407494786370250494655920e-01 7.61546840667602122076917936510654e-03
 	 1.03041271596045185682255507231275e-02 6.07018745407494786370250494655920e-01 3.82677127432900721082376094273059e-01 7.61546840667602122076917936510654e-03
 	 6.07018745407494786370250494655920e-01 3.82677127432900721082376094273059e-01 1.03041271596044925473734110710211e-02 7.61546840667602122076917936510654e-03
 	 1.07221902311333636831358617769183e-02 3.02819418977863463737776328343898e-01 6.86458390791003125741553958505392e-01 7.90075813312910400776356567575931e-03
 	 3.02819418977863463737776328343898e-01 6.86458390791003125741553958505392e-01 1.07221902311334105206697131507099e-02 7.90075813312910400776356567575931e-03
 	 6.86458390791003125741553958505392e-01 1.07221902311333636831358617769183e-02 3.02819418977863519248927559601725e-01 7.90075813312910400776356567575931e-03
 	 6.34588872173355300621011565453955e-02 4.79826825588268238154121547722752e-02 8.88558430223837625305804976960644e-01 8.21043953292401734733729767867771e-03
 	 4.79826825588268238154121547722752e-02 8.88558430223837625305804976960644e-01 6.34588872173355023065255409164820e-02 8.21043953292401734733729767867771e-03
 	 8.88558430223837625305804976960644e-01 6.34588872173355300621011565453955e-02 4.79826825588268446320938664939604e-02 8.21043953292401734733729767867771e-03
 	 1.49585341667801352549815874226624e-01 4.15353642843105178017459877537476e-02 8.08879294047888164342907657555770e-01 1.06316665808041440483000528161028e-02
 	 4.15353642843105178017459877537476e-02 8.08879294047888164342907657555770e-01 1.49585341667801352549815874226624e-01 1.06316665808041440483000528161028e-02
 	 8.08879294047888164342907657555770e-01 1.49585341667801352549815874226624e-01 4.15353642843104831072764682176057e-02 1.06316665808041440483000528161028e-02
 	 6.59383749931683749156974272409570e-02 1.27098301306499766871738188456220e-01 8.06963323700331858212564384302823e-01 1.45180631220521022539671207596257e-02
 	 1.27098301306499766871738188456220e-01 8.06963323700331858212564384302823e-01 6.59383749931683471601218116120435e-02 1.45180631220521022539671207596257e-02
 	 8.06963323700331858212564384302823e-01 6.59383749931683749156974272409570e-02 1.27098301306499766871738188456220e-01 1.45180631220521022539671207596257e-02
 	 4.19360344474693613747717790829483e-01 4.84133944843579896999408163082990e-02 5.32226261040948389613447488954989e-01 1.46941137319119476817297353932190e-02
 	 4.84133944843579896999408163082990e-02 5.32226261040948389613447488954989e-01 4.19360344474693613747717790829483e-01 1.46941137319119476817297353932190e-02
 	 5.32226261040948389613447488954989e-01 4.19360344474693613747717790829483e-01 4.84133944843579966388347202155273e-02 1.46941137319119476817297353932190e-02
 	 5.12859715627437540153010786525556e-02 3.85451519447595591927324676362332e-01 5.63262508989660681812949860614026e-01 1.68699111933565473742113738353510e-02
 	 3.85451519447595591927324676362332e-01 5.63262508989660681812949860614026e-01 5.12859715627437262597254630236421e-02 1.68699111933565473742113738353510e-02
 	 5.63262508989660681812949860614026e-01 5.12859715627437540153010786525556e-02 3.85451519447595591927324676362332e-01 1.68699111933565473742113738353510e-02
 	 5.92161717125940026407171501432458e-02 2.44263378740914083309121451748069e-01 6.96520449546491948744630917644827e-01 1.72191695246377389494352172505387e-02
 	 2.44263378740914083309121451748069e-01 6.96520449546491948744630917644827e-01 5.92161717125939679462476306071039e-02 1.72191695246377389494352172505387e-02
 	 6.96520449546491948744630917644827e-01 5.92161717125940026407171501432458e-02 2.44263378740914055553545836119156e-01 1.72191695246377389494352172505387e-02
 	 1.57744353652534891052638954533904e-01 1.07039102999093777013328576686035e-01 7.35216543348371276422881237522233e-01 1.74916741454707938552282087130152e-02
 	 1.07039102999093777013328576686035e-01 7.35216543348371276422881237522233e-01 1.57744353652534918808214570162818e-01 1.74916741454707938552282087130152e-02
 	 7.35216543348371276422881237522233e-01 1.57744353652534891052638954533904e-01 1.07039102999093832524479807943862e-01 1.74916741454707938552282087130152e-02
 	 2.76356371451293159058337778333225e-01 5.93958296507460237201314612320857e-02 6.64247798897960928243833222950343e-01 1.82914778002982274240917348606672e-02
 	 5.93958296507460237201314612320857e-02 6.64247798897960928243833222950343e-01 2.76356371451293103547186547075398e-01 1.82914778002982274240917348606672e-02
 	 6.64247798897960928243833222950343e-01 2.76356371451293159058337778333225e-01 5.93958296507459126978289987164317e-02 1.82914778002982274240917348606672e-02
 	 4.02925432504102898345621497355751e-01 1.11259697463848866716418228861585e-01 4.85814870032048151671233426895924e-01 2.10893555024571459355353653108978e-02
 	 1.11259697463848866716418228861585e-01 4.85814870032048151671233426895924e-01 4.02925432504102953856772728613578e-01 2.10893555024571459355353653108978e-02
 	 4.85814870032048151671233426895924e-01 4.02925432504102898345621497355751e-01 1.11259697463848949983145075748325e-01 2.10893555024571459355353653108978e-02
 	 1.44741209510670071347604448419588e-01 2.07337086661877140203458225187205e-01 6.47921703827452843960088557651034e-01 2.30291713310627581268175134709963e-02
 	 2.07337086661877140203458225187205e-01 6.47921703827452843960088557651034e-01 1.44741209510670043592028832790675e-01 2.30291713310627581268175134709963e-02
 	 6.47921703827452843960088557651034e-01 1.44741209510670071347604448419588e-01 2.07337086661877084692306993929378e-01 2.30291713310627581268175134709963e-02
 	 1.28002196170715176748444719123654e-01 3.38803971814336801759992567895097e-01 5.33193832014948077002713944239076e-01 2.49341884513248727983647512473908e-02
 	 3.38803971814336801759992567895097e-01 5.33193832014948077002713944239076e-01 1.28002196170715176748444719123654e-01 2.49341884513248727983647512473908e-02
 	 5.33193832014948077002713944239076e-01 1.28002196170715176748444719123654e-01 3.38803971814336746248841336637270e-01 2.49341884513248727983647512473908e-02
 	 2.67334020957130047424499252883834e-01 1.47872285803578001894820204142889e-01 5.84793693239291978436256158602191e-01 2.59561349491033883929436143489511e-02
 	 1.47872285803578001894820204142889e-01 5.84793693239291978436256158602191e-01 2.67334020957129991913348021626007e-01 2.59561349491033883929436143489511e-02
 	 5.84793693239291978436256158602191e-01 2.67334020957130047424499252883834e-01 1.47872285803577974139244588513975e-01 2.59561349491033883929436143489511e-02
 	 3.80640391337433547924717913701897e-01 2.07421341282564897223750222110539e-01 4.11938267380001554851531864187564e-01 2.89155163062293323050777615890183e-02
 	 2.07421341282564897223750222110539e-01 4.11938267380001554851531864187564e-01 3.80640391337433492413566682444070e-01 2.89155163062293323050777615890183e-02
 	 4.11938267380001554851531864187564e-01 3.80640391337433547924717913701897e-01 2.07421341282564841712598990852712e-01 2.89155163062293323050777615890183e-02
 	 2.41019074102765712419582655456907e-01 2.67083117991290863724174187154858e-01 4.91897807905943451611818773017148e-01 3.01655996798087348842454247233036e-02
 	 2.67083117991290863724174187154858e-01 4.91897807905943451611818773017148e-01 2.41019074102765629152855808570166e-01 3.01655996798087348842454247233036e-02
 	 4.91897807905943451611818773017148e-01 2.41019074102765712419582655456907e-01 2.67083117991290808213022955897031e-01 3.01655996798087348842454247233036e-02
 	 ];
 
 	 case  20 

 	 % ALG. DEG.:   20
 	 % PTS CARD.:   78
 	 % NEG. W.  :    0
 	 % OUT PTS. :    0
 	 % M.E.INF. : 4.9e-16  

 	 xyw_bar=[
 	 7.90368134162346309967706048382752e-03 1.08926176663298908409149845510910e-02 9.81203700992046679019153998524416e-01 1.13899061489750388909136802340072e-03
 	 1.08926176663298908409149845510910e-02 9.81203700992046679019153998524416e-01 7.90368134162344748716577669256367e-03 1.13899061489750388909136802340072e-03
 	 9.81203700992046679019153998524416e-01 7.90368134162346309967706048382752e-03 1.08926176663298578811689409917562e-02 1.13899061489750388909136802340072e-03
 	 3.70261421404957180927852711249670e-03 5.68359139419127579184731757777627e-02 9.39461471844037609990607506915694e-01 1.53576708745824935808954592175724e-03
 	 5.68359139419127579184731757777627e-02 9.39461471844037609990607506915694e-01 3.70261421404960433534370167762972e-03 1.53576708745824935808954592175724e-03
 	 9.39461471844037609990607506915694e-01 3.70261421404957180927852711249670e-03 5.68359139419128203685183109428181e-02 1.53576708745824935808954592175724e-03
 	 1.09337390150228475071969569398789e-01 3.03859257334367343500458602534309e-03 8.87624017276427812461747635097709e-01 1.92468559178653620539423130253454e-03
 	 3.03859257334367343500458602534309e-03 8.87624017276427812461747635097709e-01 1.09337390150228475071969569398789e-01 1.92468559178653620539423130253454e-03
 	 8.87624017276427812461747635097709e-01 1.09337390150228475071969569398789e-01 3.03859257334371246628279550350271e-03 1.92468559178653620539423130253454e-03
 	 4.28810753769432936866046190971247e-02 9.80505702549880342466792626510141e-03 9.47313867597557868194257935101632e-01 2.50043731235591402267681715443359e-03
 	 9.80505702549880342466792626510141e-03 9.47313867597557868194257935101632e-01 4.28810753769433006254985230043530e-02 2.50043731235591402267681715443359e-03
 	 9.47313867597557868194257935101632e-01 4.28810753769432936866046190971247e-02 9.80505702549883811913744580124330e-03 2.50043731235591402267681715443359e-03
 	 1.77473257307326614390374786012217e-03 2.36693018982774450176265190748381e-01 7.61532248444152237709658948006108e-01 2.63075123618766162567061961397030e-03
 	 2.36693018982774450176265190748381e-01 7.61532248444152237709658948006108e-01 1.77473257307325660292462998768315e-03 2.63075123618766162567061961397030e-03
 	 7.61532248444152237709658948006108e-01 1.77473257307326614390374786012217e-03 2.36693018982774505687416422006208e-01 2.63075123618766162567061961397030e-03
 	 3.11814570601137097671795572750852e-01 2.54551979563882565041343930545281e-03 6.85639909603224184664327367499936e-01 3.19562561402815222336148970327940e-03
 	 2.54551979563882565041343930545281e-03 6.85639909603224184664327367499936e-01 3.11814570601137042160644341493025e-01 3.19562561402815222336148970327940e-03
 	 6.85639909603224184664327367499936e-01 3.11814570601137097671795572750852e-01 2.54551979563871766387705974921118e-03 3.19562561402815222336148970327940e-03
 	 2.99000660469557651144345555849213e-03 4.63119441155936895082589899175218e-01 5.33890552239367544018477929057553e-01 3.65176899407884582998695677247269e-03
 	 4.63119441155936895082589899175218e-01 5.33890552239367544018477929057553e-01 2.99000660469550538778094050940126e-03 3.65176899407884582998695677247269e-03
 	 5.33890552239367544018477929057553e-01 2.99000660469557651144345555849213e-03 4.63119441155936895082589899175218e-01 3.65176899407884582998695677247269e-03
 	 3.22142321349414423314883038074186e-02 5.24854182546855979607514086637821e-02 9.15300349610372987463335903157713e-01 5.32453789121690480856674554388519e-03
 	 5.24854182546855979607514086637821e-02 9.15300349610372987463335903157713e-01 3.22142321349414562092761116218753e-02 5.32453789121690480856674554388519e-03
 	 9.15300349610372987463335903157713e-01 3.22142321349414423314883038074186e-02 5.24854182546855702051757930348685e-02 5.32453789121690480856674554388519e-03
 	 1.27799603324881029531301379620345e-02 1.31379538178000160320024747306888e-01 8.55840501489511740196292066684691e-01 5.84445602838894011366033254262220e-03
 	 1.31379538178000160320024747306888e-01 8.55840501489511740196292066684691e-01 1.27799603324880717281075703795068e-02 5.84445602838894011366033254262220e-03
 	 8.55840501489511740196292066684691e-01 1.27799603324881029531301379620345e-02 1.31379538178000160320024747306888e-01 5.84445602838894011366033254262220e-03
 	 1.95430723381915305569833662957535e-01 1.40953400791673909980827872345799e-02 7.90473936538917398841874728532275e-01 7.30800598014199644103516817494892e-03
 	 1.40953400791673909980827872345799e-02 7.90473936538917398841874728532275e-01 1.95430723381915250058682431699708e-01 7.30800598014199644103516817494892e-03
 	 7.90473936538917398841874728532275e-01 1.95430723381915305569833662957535e-01 1.40953400791672955882916085101897e-02 7.30800598014199644103516817494892e-03
 	 9.77669169905016482990944837183633e-02 3.50007516628948259218567784500920e-02 8.67232331346603491084579218295403e-01 7.90633535329192740914727721701638e-03
 	 3.50007516628948259218567784500920e-02 8.67232331346603491084579218295403e-01 9.77669169905016621768822915328201e-02 7.90633535329192740914727721701638e-03
 	 8.67232331346603491084579218295403e-01 9.77669169905016482990944837183633e-02 3.50007516628948606163262979862338e-02 7.90633535329192740914727721701638e-03
 	 1.66006628441778165961029145591965e-02 3.38676752265837932576175717258593e-01 6.44722584889984240419380512321368e-01 9.65595787088912443918431449674245e-03
 	 3.38676752265837932576175717258593e-01 6.44722584889984240419380512321368e-01 1.66006628441778270044437704200391e-02 9.65595787088912443918431449674245e-03
 	 6.44722584889984240419380512321368e-01 1.66006628441778165961029145591965e-02 3.38676752265837932576175717258593e-01 9.65595787088912443918431449674245e-03
 	 4.13395082180740003341412602821947e-01 1.82025878269482105364929225288506e-02 5.68402329992311883266609129350400e-01 1.03145890038596216992106135990070e-02
 	 1.82025878269482105364929225288506e-02 5.68402329992311883266609129350400e-01 4.13395082180739947830261371564120e-01 1.03145890038596216992106135990070e-02
 	 5.68402329992311883266609129350400e-01 4.13395082180740003341412602821947e-01 1.82025878269481133919782678276533e-02 1.03145890038596216992106135990070e-02
 	 6.83293746341590430004586664836097e-02 1.08675089954566678196457019112131e-01 8.22995535411274348192023353476543e-01 1.22098385442625488594980964762726e-02
 	 1.08675089954566678196457019112131e-01 8.22995535411274348192023353476543e-01 6.83293746341590013670952430402394e-02 1.22098385442625488594980964762726e-02
 	 8.22995535411274348192023353476543e-01 6.83293746341590430004586664836097e-02 1.08675089954566608807517980039847e-01 1.22098385442625488594980964762726e-02
 	 3.92153078432324539437381361040025e-02 2.14338430054550499459509182997863e-01 7.46446262102216984146707545733079e-01 1.28386823220992402072582194705319e-02
 	 2.14338430054550499459509182997863e-01 7.46446262102216984146707545733079e-01 3.92153078432324608826320400112309e-02 1.28386823220992402072582194705319e-02
 	 7.46446262102216984146707545733079e-01 3.92153078432324539437381361040025e-02 2.14338430054550554970660414255690e-01 1.28386823220992402072582194705319e-02
 	 2.82851404387685689822973245100002e-01 4.32505997246091511421184350183466e-02 6.73897995887705159034908319881652e-01 1.46667976977237508179330305324584e-02
 	 4.32505997246091511421184350183466e-02 6.73897995887705159034908319881652e-01 2.82851404387685745334124476357829e-01 1.46667976977237508179330305324584e-02
 	 6.73897995887705159034908319881652e-01 2.82851404387685689822973245100002e-01 4.32505997246091511421184350183466e-02 1.46667976977237508179330305324584e-02
 	 1.63149078355705329235192380110675e-01 7.46324203435316663179932561433816e-02 7.62218501300762962813450940302573e-01 1.54050538623656755460311629235548e-02
 	 7.46324203435316663179932561433816e-02 7.62218501300762962813450940302573e-01 1.63149078355705356990767995739589e-01 1.54050538623656755460311629235548e-02
 	 7.62218501300762962813450940302573e-01 1.63149078355705329235192380110675e-01 7.46324203435317079513566795867519e-02 1.54050538623656755460311629235548e-02
 	 4.36290583906621953769189303784515e-01 5.16537077780689246608858411491383e-01 4.71723383126887441108010534662753e-02 1.65755938748782943870097028593591e-02
 	 5.16537077780689246608858411491383e-01 4.71723383126887441108010534662753e-02 4.36290583906622009280340535042342e-01 1.65755938748782943870097028593591e-02
 	 4.71723383126887441108010534662753e-02 4.36290583906621953769189303784515e-01 5.16537077780689246608858411491383e-01 1.65755938748782943870097028593591e-02
 	 1.23069087330845969296966302408691e-01 1.73946464302243086752497447378119e-01 7.02984448366910985583899673656560e-01 1.99892203811951919834033475353863e-02
 	 1.73946464302243086752497447378119e-01 7.02984448366910985583899673656560e-01 1.23069087330845983174754110223148e-01 1.99892203811951919834033475353863e-02
 	 7.02984448366910985583899673656560e-01 1.23069087330845969296966302408691e-01 1.73946464302243031241346216120292e-01 1.99892203811951919834033475353863e-02
 	 8.32579050810742932187125120435667e-02 2.99185227797303610053347711073002e-01 6.17556867121622055094576353440061e-01 2.02974174775074275856034944354178e-02
 	 2.99185227797303610053347711073002e-01 6.17556867121622055094576353440061e-01 8.32579050810743348520759354869369e-02 2.02974174775074275856034944354178e-02
 	 6.17556867121622055094576353440061e-01 8.32579050810742932187125120435667e-02 2.99185227797303665564498942330829e-01 2.02974174775074275856034944354178e-02
 	 3.76028943583988584098420915324823e-01 8.64379444693639392705009072415123e-02 5.37533111946647546020017216505948e-01 2.16366905057804904954910085734809e-02
 	 8.64379444693639392705009072415123e-02 5.37533111946647546020017216505948e-01 3.76028943583988528587269684066996e-01 2.16366905057804904954910085734809e-02
 	 5.37533111946647546020017216505948e-01 3.76028943583988584098420915324823e-01 8.64379444693638698815618681692285e-02 2.16366905057804904954910085734809e-02
 	 2.39866290944955318220621620639577e-01 1.30115610897138883617074611720454e-01 6.30018098157905770406728152011055e-01 2.30254773629517753286144454705209e-02
 	 1.30115610897138883617074611720454e-01 6.30018098157905770406728152011055e-01 2.39866290944955373731772851897404e-01 2.30254773629517753286144454705209e-02
 	 6.30018098157905770406728152011055e-01 2.39866290944955318220621620639577e-01 1.30115610897138911372650227349368e-01 2.30254773629517753286144454705209e-02
 	 1.37221148823993804866105961082212e-01 3.91013201302957691396500194969121e-01 4.71765649873048475981818228319753e-01 2.63234937477725451648868926213254e-02
 	 3.91013201302957691396500194969121e-01 4.71765649873048475981818228319753e-01 1.37221148823993832621681576711126e-01 2.63234937477725451648868926213254e-02
 	 4.71765649873048475981818228319753e-01 1.37221148823993804866105961082212e-01 3.91013201302957691396500194969121e-01 2.63234937477725451648868926213254e-02
 	 1.89932130608380905778531655414554e-01 2.50321346008893164114539331421838e-01 5.59746523382725902351353397534695e-01 2.70480976270349934842407435553469e-02
 	 2.50321346008893164114539331421838e-01 5.59746523382725902351353397534695e-01 1.89932130608380989045258502301294e-01 2.70480976270349934842407435553469e-02
 	 5.59746523382725902351353397534695e-01 1.89932130608380905778531655414554e-01 2.50321346008893219625690562679665e-01 2.70480976270349934842407435553469e-02
 	 3.24158256395638832980665711147594e-01 1.94724488881111906790621901564009e-01 4.81117254723249176961985540401656e-01 2.89943397296065115409380297251118e-02
 	 1.94724488881111906790621901564009e-01 4.81117254723249176961985540401656e-01 3.24158256395638888491816942405421e-01 2.89943397296065115409380297251118e-02
 	 4.81117254723249176961985540401656e-01 3.24158256395638832980665711147594e-01 1.94724488881111990057348748450750e-01 2.89943397296065115409380297251118e-02
 	 3.30862912147051713152023921793443e-01 4.07445324376853323933289630076615e-01 2.61691763476095018425837679387769e-01 3.13907216215734349851196327563230e-02
 	 4.07445324376853323933289630076615e-01 2.61691763476095018425837679387769e-01 3.30862912147051657640872690535616e-01 3.13907216215734349851196327563230e-02
 	 2.61691763476095018425837679387769e-01 3.30862912147051713152023921793443e-01 4.07445324376853268422138398818788e-01 3.13907216215734349851196327563230e-02
 	 ];
 
 	 case  21 

 	 % ALG. DEG.:   21
 	 % PTS CARD.:   85
 	 % NEG. W.  :    0
 	 % OUT PTS. :    0
 	 % M.E.INF. : 4.6e-16  

 	 xyw_bar=[
 	 3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333425851918718763045e-01 1.59272439217052891313741724843567e-02
 	 1.08833811258291848766965870254353e-02 1.08437360200908869400926448633982e-02 9.78272882854079872672059536853340e-01 1.54002481041379908274779886312444e-03
 	 1.08437360200908869400926448633982e-02 9.78272882854079872672059536853340e-01 1.08833811258292056933782987471204e-02 1.54002481041379908274779886312444e-03
 	 9.78272882854079872672059536853340e-01 1.08833811258291848766965870254353e-02 1.08437360200909424512438761212252e-02 1.54002481041379908274779886312444e-03
 	 1.34483082013301280532857617799891e-01 5.89401791446114188544047607365428e-03 8.59622900072237627888682709453860e-01 2.96403821123544276849859357980677e-03
 	 5.89401791446114188544047607365428e-03 8.59622900072237627888682709453860e-01 1.34483082013301280532857617799891e-01 2.96403821123544276849859357980677e-03
 	 8.59622900072237627888682709453860e-01 1.34483082013301280532857617799891e-01 5.89401791446109157845967274624854e-03 2.96403821123544276849859357980677e-03
 	 5.60697914456082802114877949861693e-02 9.34072029515549240308303069468820e-01 9.85817903884245172463351991609670e-03 3.14257047937977103402307221813317e-03
 	 9.34072029515549240308303069468820e-01 9.85817903884245172463351991609670e-03 5.60697914456083079670634106150828e-02 3.14257047937977103402307221813317e-03
 	 9.85817903884245172463351991609670e-03 5.60697914456082802114877949861693e-02 9.34072029515549240308303069468820e-01 3.14257047937977103402307221813317e-03
 	 5.62307971638516188028766862316843e-02 1.04111689007461219458328116616030e-02 9.33358033935402287006866117735626e-01 3.32255772951118265312597799265859e-03
 	 1.04111689007461219458328116616030e-02 9.33358033935402287006866117735626e-01 5.62307971638515979861949745099992e-02 3.32255772951118265312597799265859e-03
 	 9.33358033935402287006866117735626e-01 5.62307971638516188028766862316843e-02 1.04111689007460941902571960326895e-02 3.32255772951118265312597799265859e-03
 	 4.95492250545260581873208138858899e-01 5.30139353929082006444684083135144e-03 4.99206355915448596327621544332942e-01 4.07094212767519694068374391804355e-03
 	 5.30139353929082006444684083135144e-03 4.99206355915448596327621544332942e-01 4.95492250545260581873208138858899e-01 4.07094212767519694068374391804355e-03
 	 4.99206355915448596327621544332942e-01 4.95492250545260581873208138858899e-01 5.30139353929082179917031680815853e-03 4.07094212767519694068374391804355e-03
 	 1.33413544221436503045552512958238e-01 8.56502202113258115367955269903177e-01 1.00842536653054093420678327674977e-02 4.70658038914133561303465569380933e-03
 	 8.56502202113258115367955269903177e-01 1.00842536653054093420678327674977e-02 1.33413544221436475289976897329325e-01 4.70658038914133561303465569380933e-03
 	 1.00842536653054093420678327674977e-02 1.33413544221436503045552512958238e-01 8.56502202113258115367955269903177e-01 4.70658038914133561303465569380933e-03
 	 2.37788853273795369736021143580729e-01 7.53553830034851435115683671028819e-01 8.65731669135316739271956976153888e-03 5.18588023962721512277740387730773e-03
 	 7.53553830034851435115683671028819e-01 8.65731669135316739271956976153888e-03 2.37788853273795397491596759209642e-01 5.18588023962721512277740387730773e-03
 	 8.65731669135316739271956976153888e-03 2.37788853273795369736021143580729e-01 7.53553830034851435115683671028819e-01 5.18588023962721512277740387730773e-03
 	 2.36675822687569759050774109709891e-01 1.04315998683223633974881394692602e-02 7.52892577444107868878120370936813e-01 6.05317227294636852480413935495562e-03
 	 1.04315998683223633974881394692602e-02 7.52892577444107868878120370936813e-01 2.36675822687569814561925340967719e-01 6.05317227294636852480413935495562e-03
 	 7.52892577444107868878120370936813e-01 2.36675822687569759050774109709891e-01 1.04315998683223720711055193532957e-02 6.05317227294636852480413935495562e-03
 	 3.59237431543587948379325780479121e-01 9.77441311908519919193771841037233e-03 6.30988155337326772631456606177380e-01 6.49737315478060399598358998218828e-03
 	 9.77441311908519919193771841037233e-03 6.30988155337326772631456606177380e-01 3.59237431543588003890477011736948e-01 6.49737315478060399598358998218828e-03
 	 6.30988155337326772631456606177380e-01 3.59237431543587948379325780479121e-01 9.77441311908527898921761334349867e-03 6.49737315478060399598358998218828e-03
 	 3.62123795542131587943401882512262e-01 6.27482731394552795123331634385977e-01 1.03934730633156169332664831017610e-02 6.97332416543433673133955963407971e-03
 	 6.27482731394552795123331634385977e-01 1.03934730633156169332664831017610e-02 3.62123795542131587943401882512262e-01 6.97332416543433673133955963407971e-03
 	 1.03934730633156169332664831017610e-02 3.62123795542131587943401882512262e-01 6.27482731394552795123331634385977e-01 6.97332416543433673133955963407971e-03
 	 5.47064625266307918982633395899029e-02 8.93376258603483019982149926363491e-01 5.19172788698861742417989262321498e-02 7.20587558934637339297379909908159e-03
 	 8.93376258603483019982149926363491e-01 5.19172788698861742417989262321498e-02 5.47064625266308057760511474043597e-02 7.20587558934637339297379909908159e-03
 	 5.19172788698861742417989262321498e-02 5.47064625266307918982633395899029e-02 8.93376258603483019982149926363491e-01 7.20587558934637339297379909908159e-03
 	 1.28497845564646773208750119010801e-01 3.68128421478705475378134792663332e-02 8.34689312287482665375648593908409e-01 8.36501282887073226157692573679014e-03
 	 3.68128421478705475378134792663332e-02 8.34689312287482665375648593908409e-01 1.28497845564646828719901350268628e-01 8.36501282887073226157692573679014e-03
 	 8.34689312287482665375648593908409e-01 1.28497845564646773208750119010801e-01 3.68128421478705614156012870807899e-02 8.36501282887073226157692573679014e-03
 	 1.32045621485988418086776619020384e-01 8.16750542291406311790069594280794e-01 5.12038362226052701231537866988219e-02 1.02265963368377808817566432253443e-02
 	 8.16750542291406311790069594280794e-01 5.12038362226052701231537866988219e-02 1.32045621485988418086776619020384e-01 1.02265963368377808817566432253443e-02
 	 5.12038362226052701231537866988219e-02 1.32045621485988418086776619020384e-01 8.16750542291406311790069594280794e-01 1.02265963368377808817566432253443e-02
 	 4.80002887465515781784120008524042e-01 3.46233167655783727334473098835588e-02 4.85373795768905824665750969870714e-01 1.17409287365970831096273485627535e-02
 	 3.46233167655783727334473098835588e-02 4.85373795768905824665750969870714e-01 4.80002887465515781784120008524042e-01 1.17409287365970831096273485627535e-02
 	 4.85373795768905824665750969870714e-01 4.80002887465515781784120008524042e-01 3.46233167655783935501290216052439e-02 1.17409287365970831096273485627535e-02
 	 2.35698652486183879428693899171776e-01 7.17639780294608065069894564658171e-01 4.66615672192079999902603049122263e-02 1.21832494441335686258254256131295e-02
 	 7.17639780294608065069894564658171e-01 4.66615672192079999902603049122263e-02 2.35698652486183934939845130429603e-01 1.21832494441335686258254256131295e-02
 	 4.66615672192079999902603049122263e-02 2.35698652486183879428693899171776e-01 7.17639780294608176092197027173825e-01 1.21832494441335686258254256131295e-02
 	 1.21235453505120910344672324754356e-01 9.87172892132079649041287439104053e-02 7.80047257281671124751198931335239e-01 1.28654894748754123484646427755251e-02
 	 9.87172892132079649041287439104053e-02 7.80047257281671124751198931335239e-01 1.21235453505120882589096709125442e-01 1.28654894748754123484646427755251e-02
 	 7.80047257281671124751198931335239e-01 1.21235453505120910344672324754356e-01 9.87172892132079649041287439104053e-02 1.28654894748754123484646427755251e-02
 	 2.26921312534569846741661081068742e-01 5.33826956732506574532237664243439e-02 7.19695991792179468049539536878001e-01 1.30952982620168266458815509167835e-02
 	 5.33826956732506574532237664243439e-02 7.19695991792179468049539536878001e-01 2.26921312534569930008387927955482e-01 1.30952982620168266458815509167835e-02
 	 7.19695991792179468049539536878001e-01 2.26921312534569846741661081068742e-01 5.33826956732506852087993820532574e-02 1.30952982620168266458815509167835e-02
 	 3.52725956002229834584937862018705e-01 5.93956020138616369585804477537749e-01 5.33180238591538513404088917013723e-02 1.40867190533422448339706534170546e-02
 	 5.93956020138616369585804477537749e-01 5.33180238591538513404088917013723e-02 3.52725956002229779073786630760878e-01 1.40867190533422448339706534170546e-02
 	 5.33180238591538513404088917013723e-02 3.52725956002229834584937862018705e-01 5.93956020138616258563502015022095e-01 1.40867190533422448339706534170546e-02
 	 3.46394760980635263436511195322964e-01 5.11343671910869101981944595536334e-02 6.02470871828277854120869960752316e-01 1.41840307195744228657696694995138e-02
 	 5.11343671910869101981944595536334e-02 6.02470871828277854120869960752316e-01 3.46394760980635263436511195322964e-01 1.41840307195744228657696694995138e-02
 	 6.02470871828277854120869960752316e-01 3.46394760980635263436511195322964e-01 5.11343671910868824426188439247198e-02 1.41840307195744228657696694995138e-02
 	 2.08545376179994440546394685043197e-01 1.26661874088773246382899628770247e-01 6.64792749731232257559554454928730e-01 1.72208315676023321350029959830863e-02
 	 1.26661874088773246382899628770247e-01 6.64792749731232257559554454928730e-01 2.08545376179994468301970300672110e-01 1.72208315676023321350029959830863e-02
 	 6.64792749731232257559554454928730e-01 2.08545376179994440546394685043197e-01 1.26661874088773301894050860028074e-01 1.72208315676023321350029959830863e-02
 	 1.16641181787498937616476268885890e-01 1.93583513707742599363470503703866e-01 6.89775304504758435264477611781331e-01 1.74136239593561253924391962755180e-02
 	 1.93583513707742599363470503703866e-01 6.89775304504758435264477611781331e-01 1.16641181787498937616476268885890e-01 1.74136239593561253924391962755180e-02
 	 6.89775304504758435264477611781331e-01 1.16641181787498937616476268885890e-01 1.93583513707742627119046119332779e-01 1.74136239593561253924391962755180e-02
 	 3.01681731659511120913208515048609e-01 5.79485561285726924474204224679852e-01 1.18832707054762010123738491529366e-01 1.80836755436484704351318697490569e-02
 	 5.79485561285726924474204224679852e-01 1.18832707054762010123738491529366e-01 3.01681731659511065402057283790782e-01 1.80836755436484704351318697490569e-02
 	 1.18832707054762010123738491529366e-01 3.01681731659511120913208515048609e-01 5.79485561285726813451901762164198e-01 1.80836755436484704351318697490569e-02
 	 3.59766505094364252848038177035050e-01 4.65449578776416461600717866531340e-01 1.74783916129219285551243956433609e-01 1.91491387666732279515535708469542e-02
 	 4.65449578776416461600717866531340e-01 1.74783916129219285551243956433609e-01 3.59766505094364252848038177035050e-01 1.91491387666732279515535708469542e-02
 	 1.74783916129219285551243956433609e-01 3.59766505094364252848038177035050e-01 4.65449578776416461600717866531340e-01 1.91491387666732279515535708469542e-02
 	 3.17727894779720854145921293820720e-01 1.21922765864560744830669136717916e-01 5.60349339355718401023409569461364e-01 1.92278192706125047872767197532085e-02
 	 1.21922765864560744830669136717916e-01 5.60349339355718401023409569461364e-01 3.17727894779720854145921293820720e-01 1.92278192706125047872767197532085e-02
 	 5.60349339355718401023409569461364e-01 3.17727894779720854145921293820720e-01 1.21922765864560744830669136717916e-01 1.92278192706125047872767197532085e-02
 	 4.50599237708166233673523493052926e-01 9.80260802306452266874359224857471e-02 4.51374682061188525761252776646870e-01 2.00316767614215317239168001606231e-02
 	 9.80260802306452266874359224857471e-02 4.51374682061188525761252776646870e-01 4.50599237708166289184674724310753e-01 2.00316767614215317239168001606231e-02
 	 4.51374682061188525761252776646870e-01 4.50599237708166233673523493052926e-01 9.80260802306451850540724990423769e-02 2.00316767614215317239168001606231e-02
 	 2.66780208878214575296539123883122e-01 3.09304671639405781569109876727453e-01 4.23915119482379698645502230647253e-01 2.07651228614931342153493432078903e-02
 	 3.09304671639405781569109876727453e-01 4.23915119482379698645502230647253e-01 2.66780208878214519785387892625295e-01 2.07651228614931342153493432078903e-02
 	 4.23915119482379698645502230647253e-01 2.66780208878214575296539123883122e-01 3.09304671639405726057958645469625e-01 2.07651228614931342153493432078903e-02
 	 2.14913322085512142889385245325684e-01 2.23615271231608347957120486171334e-01 5.61471406682879425886767421616241e-01 2.38150527063891379042637197471777e-02
 	 2.23615271231608347957120486171334e-01 5.61471406682879425886767421616241e-01 2.14913322085512170644960860954598e-01 2.38150527063891379042637197471777e-02
 	 5.61471406682879425886767421616241e-01 2.14913322085512142889385245325684e-01 2.23615271231608431223847333058075e-01 2.38150527063891379042637197471777e-02
 	 3.41820828914985830326855875682668e-01 2.03059575102732176032915845098614e-01 4.55119595982282021395803894847631e-01 2.39076465631621569440934393924181e-02
 	 2.03059575102732176032915845098614e-01 4.55119595982282021395803894847631e-01 3.41820828914985774815704644424841e-01 2.39076465631621569440934393924181e-02
 	 4.55119595982282021395803894847631e-01 3.41820828914985830326855875682668e-01 2.03059575102732148277340229469701e-01 2.39076465631621569440934393924181e-02
 	 ];
 
 	 case  22 

 	 % ALG. DEG.:   22
 	 % PTS CARD.:   93
 	 % NEG. W.  :    0
 	 % OUT PTS. :    0
 	 % M.E.INF. : 7.6e-16  

 	 xyw_bar=[
 	 1.17033848378872841304587648636470e-03 4.78536918807590805347018658721936e-02 9.50975969635452167416644897457445e-01 9.75412040939737322829683474623153e-04
 	 4.78536918807590805347018658721936e-02 9.50975969635452167416644897457445e-01 1.17033848378877980422885229927488e-03 9.75412040939737322829683474623153e-04
 	 9.50975969635452167416644897457445e-01 1.17033848378872841304587648636470e-03 4.78536918807591013513835775938787e-02 9.75412040939737322829683474623153e-04
 	 1.09943655740260652631246074406590e-02 9.10867175372020998480593334534205e-03 9.79896962672253679649259083817014e-01 1.31044522114979024654535244565068e-03
 	 9.10867175372020998480593334534205e-03 9.79896962672253679649259083817014e-01 1.09943655740261103659349828376435e-02 1.31044522114979024654535244565068e-03
 	 9.79896962672253679649259083817014e-01 1.09943655740260652631246074406590e-02 9.10867175372025508761630874232651e-03 1.31044522114979024654535244565068e-03
 	 1.06816925225288752131225233910072e-03 2.02709063387925630950192612544924e-01 7.96222767359821559374211119575193e-01 1.81938408464920133890407871746220e-03
 	 2.02709063387925630950192612544924e-01 7.96222767359821559374211119575193e-01 1.06816925225283743117188350879587e-03 1.81938408464920133890407871746220e-03
 	 7.96222767359821559374211119575193e-01 1.06816925225288752131225233910072e-03 2.02709063387925547683465765658184e-01 1.81938408464920133890407871746220e-03
 	 2.21228350592110340616658348267265e-03 4.21769070671465406974220968550071e-01 5.76018645822613528650890657445416e-01 2.48538971789834164244359371309656e-03
 	 4.21769070671465406974220968550071e-01 5.76018645822613528650890657445416e-01 2.21228350592106437488837400451303e-03 2.48538971789834164244359371309656e-03
 	 5.76018645822613528650890657445416e-01 2.21228350592110340616658348267265e-03 4.21769070671465351463069737292244e-01 2.48538971789834164244359371309656e-03
 	 5.72025987162534596541796361179877e-02 8.56087099920958480292476622253162e-03 9.34236530284537014523493780870922e-01 2.79430576242717648022262366680479e-03
 	 8.56087099920958480292476622253162e-03 9.34236530284537014523493780870922e-01 5.72025987162534388374979243963026e-02 2.79430576242717648022262366680479e-03
 	 9.34236530284537014523493780870922e-01 5.72025987162534596541796361179877e-02 8.56087099920952582232658301109041e-03 2.79430576242717648022262366680479e-03
 	 3.53024211615276806153218558392837e-01 6.09424125055589051280779955277467e-03 6.40881547134167228740864175051684e-01 3.54814857201149989884836344344876e-03
 	 6.09424125055589051280779955277467e-03 6.40881547134167228740864175051684e-01 3.53024211615276861664369789650664e-01 3.54814857201149989884836344344876e-03
 	 6.40881547134167228740864175051684e-01 3.53024211615276806153218558392837e-01 6.09424125055596510591726655547973e-03 3.54814857201149989884836344344876e-03
 	 1.36788957839153818563104891836701e-01 7.84998062848008687497625146534119e-03 8.55361061532366129256388376234099e-01 3.77862665631837528826264538395208e-03
 	 7.84998062848008687497625146534119e-03 8.55361061532366129256388376234099e-01 1.36788957839153790807529276207788e-01 3.77862665631837528826264538395208e-03
 	 8.55361061532366129256388376234099e-01 1.36788957839153818563104891836701e-01 7.84998062848005218050673192919930e-03 3.77862665631837528826264538395208e-03
 	 2.41209563760037715152151349684573e-01 7.02726355172032801643666743984795e-03 7.51763172688241887442472943803295e-01 4.03826628503974396910969346663478e-03
 	 7.02726355172032801643666743984795e-03 7.51763172688241887442472943803295e-01 2.41209563760037770663302580942400e-01 4.03826628503974396910969346663478e-03
 	 7.51763172688241887442472943803295e-01 2.41209563760037715152151349684573e-01 7.02726355172039740537570651213173e-03 4.03826628503974396910969346663478e-03
 	 1.16737477866878831822550210972622e-02 1.12641216429535215293178396223084e-01 8.75685035783776988260740381520009e-01 4.63321359302336800017885209967972e-03
 	 1.12641216429535215293178396223084e-01 8.75685035783776988260740381520009e-01 1.16737477866878380794446457002778e-02 4.63321359302336800017885209967972e-03
 	 8.75685035783776988260740381520009e-01 1.16737477866878831822550210972622e-02 1.12641216429535132026451549336343e-01 4.63321359302336800017885209967972e-03
 	 3.01148996347992969901952164946124e-02 4.60558965203824444145652705628891e-02 9.23829203844818258595239512942499e-01 4.93072634030214482919740603961145e-03
 	 4.60558965203824444145652705628891e-02 9.23829203844818258595239512942499e-01 3.01148996347992969901952164946124e-02 4.93072634030214482919740603961145e-03
 	 9.23829203844818258595239512942499e-01 3.01148996347992969901952164946124e-02 4.60558965203824444145652705628891e-02 4.93072634030214482919740603961145e-03
 	 4.56500154387454060156414925586432e-01 1.14379077070311217789733149174936e-02 5.32061937905514814595164807542460e-01 6.41570399077746791188792130355978e-03
 	 1.14379077070311217789733149174936e-02 5.32061937905514814595164807542460e-01 4.56500154387454060156414925586432e-01 6.41570399077746791188792130355978e-03
 	 5.32061937905514814595164807542460e-01 4.56500154387454060156414925586432e-01 1.14379077070311252484202668711077e-02 6.41570399077746791188792130355978e-03
 	 1.18075946189018649024049523177382e-02 3.04996921690560307549588969777687e-01 6.83195483690537841425793885719031e-01 6.64842504414692777453543826027271e-03
 	 3.04996921690560307549588969777687e-01 6.83195483690537841425793885719031e-01 1.18075946189017955134659132454544e-02 6.64842504414692777453543826027271e-03
 	 6.83195483690537841425793885719031e-01 1.18075946189018649024049523177382e-02 3.04996921690560307549588969777687e-01 6.64842504414692777453543826027271e-03
 	 9.61505587301470371341949316956743e-02 4.26352318690261661293661177296599e-02 8.61214209400826824492014566203579e-01 7.93735445568833665463959192720722e-03
 	 4.26352318690261661293661177296599e-02 8.61214209400826824492014566203579e-01 9.61505587301469955008315082523040e-02 7.93735445568833665463959192720722e-03
 	 8.61214209400826824492014566203579e-01 9.61505587301470371341949316956743e-02 4.26352318690261383737905021007464e-02 7.93735445568833665463959192720722e-03
 	 3.06376574992079410419965057599256e-02 1.96990580025709072220507778183674e-01 7.72371762475082990206942668010015e-01 9.39182567120321228071322394725939e-03
 	 1.96990580025709072220507778183674e-01 7.72371762475082990206942668010015e-01 3.06376574992079930837007850641385e-02 9.39182567120321228071322394725939e-03
 	 7.72371762475082990206942668010015e-01 3.06376574992079410419965057599256e-02 1.96990580025709072220507778183674e-01 9.39182567120321228071322394725939e-03
 	 1.94815244430806083775209458508471e-01 3.86641674378723973481974951482698e-02 7.66520588131321511937699142436031e-01 9.76658462379095121397298839838186e-03
 	 3.86641674378723973481974951482698e-02 7.66520588131321511937699142436031e-01 1.94815244430806111530785074137384e-01 9.76658462379095121397298839838186e-03
 	 7.66520588131321511937699142436031e-01 1.94815244430806083775209458508471e-01 3.86641674378724042870913990554982e-02 9.76658462379095121397298839838186e-03
 	 6.03655030702410985710137936166575e-02 1.06610220689147924821860158317577e-01 8.33024276240610928034868720715167e-01 9.93931223641925792877671597125300e-03
 	 1.06610220689147924821860158317577e-01 8.33024276240610928034868720715167e-01 6.03655030702411332654833131527994e-02 9.93931223641925792877671597125300e-03
 	 8.33024276240610928034868720715167e-01 6.03655030702410985710137936166575e-02 1.06610220689147966455223581760947e-01 9.93931223641925792877671597125300e-03
 	 3.15090899482059372260778218333144e-01 3.60898201743391419671347364328540e-02 6.48819280343601478833193141326774e-01 1.07857179503410734178192598164969e-02
 	 3.60898201743391419671347364328540e-02 6.48819280343601478833193141326774e-01 3.15090899482059372260778218333144e-01 1.07857179503410734178192598164969e-02
 	 6.48819280343601478833193141326774e-01 3.15090899482059372260778218333144e-01 3.60898201743391489060286403400823e-02 1.07857179503410734178192598164969e-02
 	 4.11358367215817199991079178289510e-01 5.56390353961855654851831332052825e-01 3.22512788223271451570894896576647e-02 1.11987138171200303893293437340617e-02
 	 5.56390353961855654851831332052825e-01 3.22512788223271451570894896576647e-02 4.11358367215817199991079178289510e-01 1.11987138171200303893293437340617e-02
 	 3.22512788223271451570894896576647e-02 4.11358367215817199991079178289510e-01 5.56390353961855654851831332052825e-01 1.11987138171200303893293437340617e-02
 	 1.44020350630489235577158524392871e-01 9.77155901845127572435956153640291e-02 7.58264059184998062690397091500927e-01 1.35924492506738405633859301246957e-02
 	 9.77155901845127572435956153640291e-02 7.58264059184998062690397091500927e-01 1.44020350630489124554856061877217e-01 1.35924492506738405633859301246957e-02
 	 7.58264059184998062690397091500927e-01 1.44020350630489235577158524392871e-01 9.77155901845127017324443841062021e-02 1.35924492506738405633859301246957e-02
 	 4.20757709102121901789672619997873e-01 5.55288374894526703950958790301229e-02 5.23713453408425344548504654085264e-01 1.41072873131625007442346841912695e-02
 	 5.55288374894526703950958790301229e-02 5.23713453408425344548504654085264e-01 4.20757709102121957300823851255700e-01 1.41072873131625007442346841912695e-02
 	 5.23713453408425344548504654085264e-01 4.20757709102121901789672619997873e-01 5.55288374894527536618227259168634e-02 1.41072873131625007442346841912695e-02
 	 6.09498995807896612464382712914812e-02 2.92056016141072904979125723912148e-01 6.46994084278137426835542100889143e-01 1.43454989859199497775898990425958e-02
 	 2.92056016141072904979125723912148e-01 6.46994084278137426835542100889143e-01 6.09498995807896681853321751987096e-02 1.43454989859199497775898990425958e-02
 	 6.46994084278137426835542100889143e-01 6.09498995807896612464382712914812e-02 2.92056016141072904979125723912148e-01 1.43454989859199497775898990425958e-02
 	 9.69351064455874905645416106381163e-02 1.84224156891908047439798679079104e-01 7.18840736662504475873447518097237e-01 1.50331446811143264186139134608311e-02
 	 1.84224156891908047439798679079104e-01 7.18840736662504475873447518097237e-01 9.69351064455874489311781871947460e-02 1.50331446811143264186139134608311e-02
 	 7.18840736662504475873447518097237e-01 9.69351064455874905645416106381163e-02 1.84224156891908019684223063450190e-01 1.50331446811143264186139134608311e-02
 	 2.53783465620646964122641975336592e-01 9.22286849014402332347373203447205e-02 6.53987849477912774887045088689774e-01 1.58376778038056276187717230641283e-02
 	 9.22286849014402332347373203447205e-02 6.53987849477912774887045088689774e-01 2.53783465620647019633793206594419e-01 1.58376778038056276187717230641283e-02
 	 6.53987849477912774887045088689774e-01 2.53783465620646964122641975336592e-01 9.22286849014402609903129359736340e-02 1.58376778038056276187717230641283e-02
 	 3.48867101402162760326319812520524e-01 1.16235552070709735605902324095950e-01 5.34897346527127615090080325899180e-01 1.71145055482667535207141895625682e-02
 	 1.16235552070709735605902324095950e-01 5.34897346527127615090080325899180e-01 3.48867101402162704815168581262697e-01 1.71145055482667535207141895625682e-02
 	 5.34897346527127615090080325899180e-01 3.48867101402162760326319812520524e-01 1.16235552070709624583599861580296e-01 1.71145055482667535207141895625682e-02
 	 1.90581558653074217835055037539860e-01 1.72148246666744941801852064600098e-01 6.37270194680180868118668513488956e-01 1.79712911006388413592826935882840e-02
 	 1.72148246666744941801852064600098e-01 6.37270194680180868118668513488956e-01 1.90581558653074134568328190653119e-01 1.79712911006388413592826935882840e-02
 	 6.37270194680180868118668513488956e-01 1.90581558653074217835055037539860e-01 1.72148246666744914046276448971184e-01 1.79712911006388413592826935882840e-02
 	 9.69881143348128027525945071829483e-02 3.96552131508626504174941374003538e-01 5.06459754156560748583615350071341e-01 1.85450331969024689859892873755598e-02
 	 3.96552131508626504174941374003538e-01 5.06459754156560748583615350071341e-01 9.69881143348128027525945071829483e-02 1.85450331969024689859892873755598e-02
 	 5.06459754156560748583615350071341e-01 9.69881143348128027525945071829483e-02 3.96552131508626448663790142745711e-01 1.85450331969024689859892873755598e-02
 	 1.41266334630631706925640855843085e-01 2.75379650245183016377126250517904e-01 5.83354015124185165674930431123357e-01 1.93981112463580615545843954805605e-02
 	 2.75379650245183016377126250517904e-01 5.83354015124185165674930431123357e-01 1.41266334630631762436792087100912e-01 1.93981112463580615545843954805605e-02
 	 5.83354015124185165674930431123357e-01 1.41266334630631706925640855843085e-01 2.75379650245183127399428713033558e-01 1.93981112463580615545843954805605e-02
 	 4.12018101979952200419887731186463e-01 1.68573884290088321646905455963861e-01 4.19408013729959394666479965962935e-01 1.95236705864015071265615830498064e-02
 	 1.68573884290088321646905455963861e-01 4.19408013729959394666479965962935e-01 4.12018101979952255931038962444291e-01 1.95236705864015071265615830498064e-02
 	 4.19408013729959394666479965962935e-01 4.12018101979952200419887731186463e-01 1.68573884290088404913632302850601e-01 1.95236705864015071265615830498064e-02
 	 2.76209339065321413109188597445609e-01 1.99199356373184033319745367407450e-01 5.24591304561494498059914803889114e-01 2.06255319696370398308005178478197e-02
 	 1.99199356373184033319745367407450e-01 5.24591304561494498059914803889114e-01 2.76209339065321524131491059961263e-01 2.06255319696370398308005178478197e-02
 	 5.24591304561494498059914803889114e-01 2.76209339065321413109188597445609e-01 1.99199356373184088830896598665277e-01 2.06255319696370398308005178478197e-02
 	 2.18610614672497399757489233707020e-01 3.05685651047730433393923021867522e-01 4.75703734279772139093012128796545e-01 2.13138224201828534176694773805139e-02
 	 3.05685651047730433393923021867522e-01 4.75703734279772139093012128796545e-01 2.18610614672497483024216080593760e-01 2.13138224201828534176694773805139e-02
 	 4.75703734279772139093012128796545e-01 2.18610614672497399757489233707020e-01 3.05685651047730488905074253125349e-01 2.13138224201828534176694773805139e-02
 	 3.36799046311846372869780452674604e-01 2.69814345781991038464298071630765e-01 3.93386607906162533154770244436804e-01 2.35277531670229217430900803265104e-02
 	 2.69814345781991038464298071630765e-01 3.93386607906162533154770244436804e-01 3.36799046311846428380931683932431e-01 2.35277531670229217430900803265104e-02
 	 3.93386607906162533154770244436804e-01 3.36799046311846372869780452674604e-01 2.69814345781991093975449302888592e-01 2.35277531670229217430900803265104e-02
 	 ];
 
 	 case  23 

 	 % ALG. DEG.:   23
 	 % PTS CARD.:  100
 	 % NEG. W.  :    0
 	 % OUT PTS. :    0
 	 % M.E.INF. : 6.7e-16  

 	 xyw_bar=[
 	 3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333425851918718763045e-01 2.51388714602936956532541756814680e-02
 	 9.41394903300460904327362499088849e-03 8.64644445189447809207994311009315e-03 9.81939606515100926742434239713475e-01 1.06611768250602771158919157556966e-03
 	 8.64644445189447809207994311009315e-03 9.81939606515100926742434239713475e-01 9.41394903300463159467881268938072e-03 1.06611768250602771158919157556966e-03
 	 9.81939606515100926742434239713475e-01 9.41394903300460904327362499088849e-03 8.64644445189446421429213529563640e-03 1.06611768250602771158919157556966e-03
 	 4.82410218908890808548611817130336e-04 1.08821420580560865309927010002866e-01 8.90696169200530185605657607084140e-01 1.10987315055489140641631351513752e-03
 	 1.08821420580560865309927010002866e-01 8.90696169200530185605657607084140e-01 4.82410218908935206627575098536909e-04 1.10987315055489140641631351513752e-03
 	 8.90696169200530185605657607084140e-01 4.82410218908890808548611817130336e-04 1.08821420580560920821078241260693e-01 1.10987315055489140641631351513752e-03
 	 4.96335565629436092915938161240774e-01 5.03294735930981729232769339432707e-01 3.69698439582122340141268068691716e-04 1.62962395564562878558234615411493e-03
 	 5.03294735930981729232769339432707e-01 3.69698439582122340141268068691716e-04 4.96335565629436148427089392498601e-01 1.62962395564562878558234615411493e-03
 	 3.69698439582122340141268068691716e-04 4.96335565629436092915938161240774e-01 5.03294735930981840255071801948361e-01 1.62962395564562878558234615411493e-03
 	 8.61452759024325040448921697588958e-03 4.49648589409331542188219543731975e-02 9.46420613468823557212772357161157e-01 2.20529169313777488950489846786240e-03
 	 4.49648589409331542188219543731975e-02 9.46420613468823557212772357161157e-01 8.61452759024328162951178455841728e-03 2.20529169313777488950489846786240e-03
 	 9.46420613468823557212772357161157e-01 8.61452759024325040448921697588958e-03 4.49648589409331889132914739093394e-02 2.20529169313777488950489846786240e-03
 	 4.91006634687758056201367651283363e-02 8.24249708242969815474854300418883e-03 9.42656839448794414693111320957541e-01 2.31755431979663913816258613564969e-03
 	 8.24249708242969815474854300418883e-03 9.42656839448794414693111320957541e-01 4.91006634687758403146062846644782e-02 2.31755431979663913816258613564969e-03
 	 9.42656839448794414693111320957541e-01 4.91006634687758056201367651283363e-02 8.24249708242977968675191391412227e-03 2.31755431979663913816258613564969e-03
 	 2.66041607735387425695550156490299e-03 2.94704482021720892692684401481529e-01 7.02635101900925151952037595037837e-01 2.32163870161540206171757461106608e-03
 	 2.94704482021720892692684401481529e-01 7.02635101900925151952037595037837e-01 2.66041607735401086642923473846167e-03 2.32163870161540206171757461106608e-03
 	 7.02635101900925151952037595037837e-01 2.66041607735387425695550156490299e-03 2.94704482021720948203835632739356e-01 2.32163870161540206171757461106608e-03
 	 3.30984469269412107106376197407371e-01 3.82066849179892769131083873901389e-03 6.65194862238788875430373082053848e-01 3.00366151139556600280222653509554e-03
 	 3.82066849179892769131083873901389e-03 6.65194862238788875430373082053848e-01 3.30984469269412162617527428665198e-01 3.00366151139556600280222653509554e-03
 	 6.65194862238788875430373082053848e-01 3.30984469269412107106376197407371e-01 3.82066849179901746325072053878102e-03 3.00366151139556600280222653509554e-03
 	 1.18154505921019392644666368141770e-01 8.41315741466065573905819263700323e-03 8.73432336664319963759339771058876e-01 3.54762192353746518697077405590790e-03
 	 8.41315741466065573905819263700323e-03 8.73432336664319963759339771058876e-01 1.18154505921019392644666368141770e-01 3.54762192353746518697077405590790e-03
 	 8.73432336664319963759339771058876e-01 1.18154505921019392644666368141770e-01 8.41315741466064359599386079935357e-03 3.54762192353746518697077405590790e-03
 	 2.14073097879028895196285020574578e-01 7.20539757650225185653392045992405e-03 7.78721504544468867692330604768358e-01 3.80265163654702493395420859201295e-03
 	 7.20539757650225185653392045992405e-03 7.78721504544468867692330604768358e-01 2.14073097879028839685133789316751e-01 3.80265163654702493395420859201295e-03
 	 7.78721504544468867692330604768358e-01 2.14073097879028895196285020574578e-01 7.20539757650223711138437465706374e-03 3.80265163654702493395420859201295e-03
 	 4.45774128773336539288685287374392e-02 4.30263570230853334419585110026674e-02 9.12396230099581040384748575888807e-01 4.73432470265694373828813112936587e-03
 	 4.30263570230853334419585110026674e-02 9.12396230099581040384748575888807e-01 4.45774128773336331121868170157541e-02 4.73432470265694373828813112936587e-03
 	 9.12396230099581040384748575888807e-01 4.45774128773336539288685287374392e-02 4.30263570230853056863828953737539e-02 4.73432470265694373828813112936587e-03
 	 1.91431395153490890859160344916745e-01 3.11692698898602960078108026209520e-02 7.77399334956648813133028852462303e-01 4.78204552944848284429246731974672e-03
 	 3.11692698898602960078108026209520e-02 7.77399334956648813133028852462303e-01 1.91431395153490946370311576174572e-01 4.78204552944848284429246731974672e-03
 	 7.77399334956648813133028852462303e-01 1.91431395153490890859160344916745e-01 3.11692698898602960078108026209520e-02 4.78204552944848284429246731974672e-03
 	 1.05850984567690802745643097182437e-02 1.91864903048277385400055550235265e-01 7.97549998494953560346232279698597e-01 5.05492016730007057145890314586723e-03
 	 1.91864903048277385400055550235265e-01 7.97549998494953560346232279698597e-01 1.05850984567690264981365544372238e-02 5.05492016730007057145890314586723e-03
 	 7.97549998494953560346232279698597e-01 1.05850984567690802745643097182437e-02 1.91864903048277357644479934606352e-01 5.05492016730007057145890314586723e-03
 	 2.54202264290056406670359478994214e-02 1.05407606480096824319581116924383e-01 8.69172167090897507257807319547283e-01 6.04520613448391707506202052968547e-03
 	 1.05407606480096824319581116924383e-01 8.69172167090897507257807319547283e-01 2.54202264290056545448237557138782e-02 6.04520613448391707506202052968547e-03
 	 8.69172167090897507257807319547283e-01 2.54202264290056406670359478994214e-02 1.05407606480096852075156732553296e-01 6.04520613448391707506202052968547e-03
 	 5.91292919761095991937338567367988e-01 1.21072892452521504308426969487300e-02 3.96599790993651835080413547984790e-01 6.31270244568983042443566233714591e-03
 	 1.21072892452521504308426969487300e-02 3.96599790993651835080413547984790e-01 5.91292919761095991937338567367988e-01 6.31270244568983042443566233714591e-03
 	 3.96599790993651835080413547984790e-01 5.91292919761095991937338567367988e-01 1.21072892452521729822478846472222e-02 6.31270244568983042443566233714591e-03
 	 1.04903656520941024332671531738015e-01 4.33709753919495621943269725306891e-02 8.51725368087109413473001495731296e-01 6.77640548713800017904818062675076e-03
 	 4.33709753919495621943269725306891e-02 8.51725368087109413473001495731296e-01 1.04903656520941024332671531738015e-01 6.77640548713800017904818062675076e-03
 	 8.51725368087109413473001495731296e-01 1.04903656520941024332671531738015e-01 4.33709753919495621943269725306891e-02 6.77640548713800017904818062675076e-03
 	 4.27217446018734792190230109554250e-01 1.66970788554706185691234310297659e-02 5.56085475125794537198942180111771e-01 7.88479242763255247505860268120159e-03
 	 1.66970788554706185691234310297659e-02 5.56085475125794537198942180111771e-01 4.27217446018734792190230109554250e-01 7.88479242763255247505860268120159e-03
 	 5.56085475125794537198942180111771e-01 4.27217446018734792190230109554250e-01 1.66970788554706706108277103339788e-02 7.88479242763255247505860268120159e-03
 	 1.77899296750879898354469332844019e-01 6.45110680237001715697786607961461e-02 7.57589635225419888442388582916465e-01 8.73690764074747973355616892376929e-03
 	 6.45110680237001715697786607961461e-02 7.57589635225419888442388582916465e-01 1.77899296750879898354469332844019e-01 8.73690764074747973355616892376929e-03
 	 7.57589635225419888442388582916465e-01 1.77899296750879898354469332844019e-01 6.45110680237002132031420842395164e-02 8.73690764074747973355616892376929e-03
 	 8.05482155825715390173868968304305e-02 1.01976993969515816740312175170402e-01 8.17474790447912713631239967071451e-01 9.43440585399333946081679158623956e-03
 	 1.01976993969515816740312175170402e-01 8.17474790447912713631239967071451e-01 8.05482155825714141172966265003197e-02 9.43440585399333946081679158623956e-03
 	 8.17474790447912713631239967071451e-01 8.05482155825715390173868968304305e-02 1.01976993969515747351373136098118e-01 9.43440585399333946081679158623956e-03
 	 3.13173112798550493107718750707136e-02 2.85342978045655892849197243776871e-01 6.83339710674489064778924785059644e-01 9.49087942781928067814689597980760e-03
 	 2.85342978045655892849197243776871e-01 6.83339710674489064778924785059644e-01 3.13173112798550423718779711634852e-02 9.49087942781928067814689597980760e-03
 	 6.83339710674489064778924785059644e-01 3.13173112798550493107718750707136e-02 2.85342978045655892849197243776871e-01 9.49087942781928067814689597980760e-03
 	 2.95446758742554271837121859789477e-01 3.28810125354330601288310731433739e-02 6.71672228722012709667410490510520e-01 1.02109626562177507880235793891188e-02
 	 3.28810125354330601288310731433739e-02 6.71672228722012709667410490510520e-01 2.95446758742554216325970628531650e-01 1.02109626562177507880235793891188e-02
 	 6.71672228722012709667410490510520e-01 2.95446758742554271837121859789477e-01 3.28810125354330184954676497000037e-02 1.02109626562177507880235793891188e-02
 	 5.41132173539479316426437094378343e-02 1.83859582702943807897000283446687e-01 7.62027199943108302093719430558849e-01 1.05192533354850435017846521645879e-02
 	 1.83859582702943807897000283446687e-01 7.62027199943108302093719430558849e-01 5.41132173539479177648559016233776e-02 1.05192533354850435017846521645879e-02
 	 7.62027199943108302093719430558849e-01 5.41132173539479316426437094378343e-02 1.83859582702943780141424667817773e-01 1.05192533354850435017846521645879e-02
 	 4.27400654280490135961123598917766e-02 4.43558986603905025525307337375125e-01 5.13700947968045995573049822269240e-01 1.16891203752775181545775495806083e-02
 	 4.43558986603905025525307337375125e-01 5.13700947968045995573049822269240e-01 4.27400654280489789016428403556347e-02 1.16891203752775181545775495806083e-02
 	 5.13700947968045995573049822269240e-01 4.27400654280490135961123598917766e-02 4.43558986603904970014156106117298e-01 1.16891203752775181545775495806083e-02
 	 1.45187675973377866567304295131180e-01 1.29795450451007293724003943680145e-01 7.25016873575614839708691761188675e-01 1.37004002794014461547034144928148e-02
 	 1.29795450451007293724003943680145e-01 7.25016873575614839708691761188675e-01 1.45187675973377894322879910760093e-01 1.37004002794014461547034144928148e-02
 	 7.25016873575614839708691761188675e-01 1.45187675973377866567304295131180e-01 1.29795450451007293724003943680145e-01 1.37004002794014461547034144928148e-02
 	 7.47139784787638483454230708957766e-02 3.29913961425314705699918249592884e-01 5.95372060095921473710234295140253e-01 1.44604512990827559909545740879366e-02
 	 3.29913961425314705699918249592884e-01 5.95372060095921473710234295140253e-01 7.47139784787638205898474552668631e-02 1.44604512990827559909545740879366e-02
 	 5.95372060095921473710234295140253e-01 7.47139784787638483454230708957766e-02 3.29913961425314705699918249592884e-01 1.44604512990827559909545740879366e-02
 	 3.81968043130448131616816453970387e-01 6.73625275151304159226839374241536e-02 5.50669429354421424704923992976546e-01 1.52313957835889229180947879171981e-02
 	 6.73625275151304159226839374241536e-02 5.50669429354421424704923992976546e-01 3.81968043130448187127967685228214e-01 1.52313957835889229180947879171981e-02
 	 5.50669429354421424704923992976546e-01 3.81968043130448131616816453970387e-01 6.73625275151304436782595530530671e-02 1.52313957835889229180947879171981e-02
 	 2.21783549821814751634718732020701e-01 6.68980984358149566304518884862773e-01 1.09235465820035626549611151858699e-01 1.54858733194458367959933653423832e-02
 	 6.68980984358149566304518884862773e-01 1.09235465820035626549611151858699e-01 2.21783549821814807145869963278528e-01 1.54858733194458367959933653423832e-02
 	 1.09235465820035626549611151858699e-01 2.21783549821814751634718732020701e-01 6.68980984358149566304518884862773e-01 1.54858733194458367959933653423832e-02
 	 2.56870811513146835025622749526519e-01 9.60491211353747958989757194103731e-02 6.47080067351478382953189338877564e-01 1.59889574365544989698051381310506e-02
 	 9.60491211353747958989757194103731e-02 6.47080067351478382953189338877564e-01 2.56870811513146835025622749526519e-01 1.59889574365544989698051381310506e-02
 	 6.47080067351478382953189338877564e-01 2.56870811513146835025622749526519e-01 9.60491211353747820211879115959164e-02 1.59889574365544989698051381310506e-02
 	 1.12751303426959964548714765442128e-01 4.23674182024581569283583348806133e-01 4.63574514548458493923277501380653e-01 1.89290359058916467238642411530236e-02
 	 4.23674182024581569283583348806133e-01 4.63574514548458493923277501380653e-01 1.12751303426959992304290381071041e-01 1.89290359058916467238642411530236e-02
 	 4.63574514548458493923277501380653e-01 1.12751303426959964548714765442128e-01 4.23674182024581624794734580063960e-01 1.89290359058916467238642411530236e-02
 	 2.10323057441033522962925417232327e-01 1.87001109672332688216300766725908e-01 6.02675832886633733309622584783938e-01 1.97480813567340596215160530846333e-02
 	 1.87001109672332688216300766725908e-01 6.02675832886633733309622584783938e-01 2.10323057441033522962925417232327e-01 1.97480813567340596215160530846333e-02
 	 6.02675832886633733309622584783938e-01 2.10323057441033522962925417232327e-01 1.87001109672332743727451997983735e-01 1.97480813567340596215160530846333e-02
 	 1.61575209179476625420690538703639e-01 2.99012948455699423266906933349674e-01 5.39411842364823979067978143575601e-01 2.06677815596659590624550872917098e-02
 	 2.99012948455699423266906933349674e-01 5.39411842364823979067978143575601e-01 1.61575209179476653176266154332552e-01 2.06677815596659590624550872917098e-02
 	 5.39411842364823979067978143575601e-01 1.61575209179476625420690538703639e-01 2.99012948455699367755755702091847e-01 2.06677815596659590624550872917098e-02
 	 3.31656221355574820641720634739613e-01 1.50616235510018708909640849924472e-01 5.17727543134406498204214130964829e-01 2.07983426647951052945728633858380e-02
 	 1.50616235510018708909640849924472e-01 5.17727543134406498204214130964829e-01 3.31656221355574820641720634739613e-01 2.07983426647951052945728633858380e-02
 	 5.17727543134406498204214130964829e-01 3.31656221355574820641720634739613e-01 1.50616235510018681154065234295558e-01 2.07983426647951052945728633858380e-02
 	 2.12294763316891205162662004113372e-01 3.84271485252512368102628670385457e-01 4.03433751430596398979133709872258e-01 2.35755748407441359515424039727804e-02
 	 3.84271485252512368102628670385457e-01 4.03433751430596398979133709872258e-01 2.12294763316891177407086388484458e-01 2.35755748407441359515424039727804e-02
 	 4.03433751430596398979133709872258e-01 2.12294763316891205162662004113372e-01 3.84271485252512423613779901643284e-01 2.35755748407441359515424039727804e-02
 	 2.74113742189350695355187781387940e-01 2.58726723002178027321917852532351e-01 4.67159534808471277322894366079709e-01 2.36918543087045081751984554330193e-02
 	 2.58726723002178027321917852532351e-01 4.67159534808471277322894366079709e-01 2.74113742189350695355187781387940e-01 2.36918543087045081751984554330193e-02
 	 4.67159534808471277322894366079709e-01 2.74113742189350695355187781387940e-01 2.58726723002178027321917852532351e-01 2.36918543087045081751984554330193e-02
 	 ];
 
 	 case  24 

 	 % ALG. DEG.:   24
 	 % PTS CARD.:  109
 	 % NEG. W.  :    0
 	 % OUT PTS. :    0
 	 % M.E.INF. : 7.8e-16  

 	 xyw_bar=[
 	 3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333425851918718763045e-01 1.06817769333868110154250885557303e-02
 	 7.27744218228642408829998089458968e-03 1.87178063161081678807129424768618e-03 9.90850777186102704696679666085402e-01 3.43072138434039877394332584970016e-04
 	 1.87178063161081678807129424768618e-03 9.90850777186102704696679666085402e-01 7.27744218228643102719388480181806e-03 3.43072138434039877394332584970016e-04
 	 9.90850777186102704696679666085402e-01 7.27744218228642408829998089458968e-03 1.87178063161087121502035302000877e-03 3.43072138434039877394332584970016e-04
 	 9.37621417401595796947688654654485e-02 2.22584177630761334901188241985892e-03 9.04012016483532798716282741224859e-01 1.22299970203740672805570177672507e-03
 	 2.22584177630761334901188241985892e-03 9.04012016483532798716282741224859e-01 9.37621417401596213281322889088187e-02 1.22299970203740672805570177672507e-03
 	 9.04012016483532798716282741224859e-01 9.37621417401595796947688654654485e-02 2.22584177630762158894839330969262e-03 1.22299970203740672805570177672507e-03
 	 2.75499567254815017069979887764930e-02 9.64825831627800223344593177898787e-01 7.62421164671822637615150597412139e-03 1.63816027364919319930791630213207e-03
 	 9.64825831627800223344593177898787e-01 7.62421164671822637615150597412139e-03 2.75499567254815502792553161270916e-02 1.63816027364919319930791630213207e-03
 	 7.62421164671822637615150597412139e-03 2.75499567254815017069979887764930e-02 9.64825831627800223344593177898787e-01 1.63816027364919319930791630213207e-03
 	 3.88491565765760812900531107061397e-02 9.99221468468047303856938157196055e-03 9.51158628738743461283888791513164e-01 2.23722594587012334893527665258262e-03
 	 9.99221468468047303856938157196055e-03 9.51158628738743461283888791513164e-01 3.88491565765760604733713989844546e-02 2.23722594587012334893527665258262e-03
 	 9.51158628738743461283888791513164e-01 3.88491565765760812900531107061397e-02 9.99221468468045742605809778069670e-03 2.23722594587012334893527665258262e-03
 	 8.45824643422994626451583144444157e-02 9.07583350374654984626943132752785e-01 7.83418528304558048347416843171231e-03 2.85750186827827190247375277465380e-03
 	 9.07583350374654984626943132752785e-01 7.83418528304558048347416843171231e-03 8.45824643422994348895826988155022e-02 2.85750186827827190247375277465380e-03
 	 7.83418528304558048347416843171231e-03 8.45824643422994626451583144444157e-02 9.07583350374654984626943132752785e-01 2.85750186827827190247375277465380e-03
 	 1.69711974261767234439801654843905e-01 8.55094769491029395303005600226243e-03 8.21737078043322521914149092481239e-01 3.68582961632744551816465161664382e-03
 	 8.55094769491029395303005600226243e-03 8.21737078043322521914149092481239e-01 1.69711974261767206684226039214991e-01 3.68582961632744551816465161664382e-03
 	 8.21737078043322521914149092481239e-01 1.69711974261767234439801654843905e-01 8.55094769491024364604925267485669e-03 3.68582961632744551816465161664382e-03
 	 2.61114259888929389763490007680957e-01 7.35523531115195392665606632931485e-03 7.31530504799918657177215663978131e-01 3.69479544205243140217453934326386e-03
 	 7.35523531115195392665606632931485e-03 7.31530504799918657177215663978131e-01 2.61114259888929334252338776423130e-01 3.69479544205243140217453934326386e-03
 	 7.31530504799918657177215663978131e-01 2.61114259888929389763490007680957e-01 7.35523531115195305929432834091131e-03 3.69479544205243140217453934326386e-03
 	 1.67919705131083785154544329998316e-01 8.24005083539038452045133453793824e-01 8.07521132987776280032221620786004e-03 3.90203200620795401243334765695181e-03
 	 8.24005083539038452045133453793824e-01 8.07521132987776280032221620786004e-03 1.67919705131083785154544329998316e-01 3.90203200620795401243334765695181e-03
 	 8.07521132987776280032221620786004e-03 1.67919705131083785154544329998316e-01 8.24005083539038452045133453793824e-01 3.90203200620795401243334765695181e-03
 	 3.66562445874964837333465084157069e-01 7.00566016949978002253951459010750e-03 6.26431893955535445961402274406282e-01 4.02806909693100186065084500341982e-03
 	 7.00566016949978002253951459010750e-03 6.26431893955535445961402274406282e-01 3.66562445874964781822313852899242e-01 4.02806909693100186065084500341982e-03
 	 6.26431893955535445961402274406282e-01 3.66562445874964837333465084157069e-01 7.00566016949971670513264143664856e-03 4.02806909693100186065084500341982e-03
 	 2.71431359408183414672066646744497e-01 7.20628297746085721975362048397074e-01 7.94034284573086335257130485842936e-03 4.52488072016167704181288655718163e-03
 	 7.20628297746085721975362048397074e-01 7.94034284573086335257130485842936e-03 2.71431359408183414672066646744497e-01 4.52488072016167704181288655718163e-03
 	 7.94034284573086335257130485842936e-03 2.71431359408183414672066646744497e-01 7.20628297746085721975362048397074e-01 4.52488072016167704181288655718163e-03
 	 4.83415122297192079159344757499639e-01 7.36119356684007599550545464239804e-03 5.09223684135967857855575857684016e-01 4.53960292626740761351422648317566e-03
 	 7.36119356684007599550545464239804e-03 5.09223684135967857855575857684016e-01 4.83415122297192079159344757499639e-01 4.53960292626740761351422648317566e-03
 	 5.09223684135967857855575857684016e-01 4.83415122297192079159344757499639e-01 7.36119356684006298507938481634483e-03 4.53960292626740761351422648317566e-03
 	 3.87957374245691544167868869408267e-01 6.04329908418651196022608473867876e-01 7.71271733565725980952265672385693e-03 4.74569544731183023261733566755538e-03
 	 6.04329908418651196022608473867876e-01 7.71271733565725980952265672385693e-03 3.87957374245691544167868869408267e-01 4.74569544731183023261733566755538e-03
 	 7.71271733565725980952265672385693e-03 3.87957374245691544167868869408267e-01 6.04329908418651196022608473867876e-01 4.74569544731183023261733566755538e-03
 	 5.20536116012078262094142644400563e-02 9.07672123610022807405073308473220e-01 4.02742647887693872021941388084088e-02 5.18053657337445849773160944096162e-03
 	 9.07672123610022807405073308473220e-01 4.02742647887693872021941388084088e-02 5.20536116012078053927325527183712e-02 5.18053657337445849773160944096162e-03
 	 4.02742647887693872021941388084088e-02 5.20536116012078262094142644400563e-02 9.07672123610022807405073308473220e-01 5.18053657337445849773160944096162e-03
 	 2.29018909213725946338868766360974e-01 8.93452564981080527006795932720706e-02 6.81635834288165987082663832552498e-01 5.31192323272322488442176435796682e-03
 	 8.93452564981080527006795932720706e-02 6.81635834288165987082663832552498e-01 2.29018909213725918583293150732061e-01 5.31192323272322488442176435796682e-03
 	 6.81635834288165987082663832552498e-01 2.29018909213725946338868766360974e-01 8.93452564981080665784674010865274e-02 5.31192323272322488442176435796682e-03
 	 9.49141955716360924411034716285940e-02 2.67967190917923619997154105476511e-02 8.78289085336571573314756733452668e-01 5.36419504296679373889222830484869e-03
 	 2.67967190917923619997154105476511e-02 8.78289085336571573314756733452668e-01 9.49141955716360508077400481852237e-02 5.36419504296679373889222830484869e-03
 	 8.78289085336571573314756733452668e-01 9.49141955716360924411034716285940e-02 2.67967190917923342441397949187376e-02 5.36419504296679373889222830484869e-03
 	 1.23228014736413052365726628067932e-01 8.35184452859199955021551886602538e-01 4.15875324043869509793580618861597e-02 7.84096594518897405112767273749341e-03
 	 8.35184452859199955021551886602538e-01 4.15875324043869509793580618861597e-02 1.23228014736413093999090051511303e-01 7.84096594518897405112767273749341e-03
 	 4.15875324043869509793580618861597e-02 1.23228014736413052365726628067932e-01 8.35184452859199955021551886602538e-01 7.84096594518897405112767273749341e-03
 	 1.74126507274859759766272304659651e-01 4.41306288863040421799333046237734e-02 7.81742863838836177237112678994890e-01 8.69940226621333627388121811918609e-03
 	 4.41306288863040421799333046237734e-02 7.81742863838836177237112678994890e-01 1.74126507274859787521847920288565e-01 8.69940226621333627388121811918609e-03
 	 7.81742863838836177237112678994890e-01 1.74126507274859759766272304659651e-01 4.41306288863040629966150163454586e-02 8.69940226621333627388121811918609e-03
 	 2.72180746005686646782351090223528e-01 3.74687219394429782415301133369212e-02 6.90350532054870402731694412068464e-01 8.84020363336787337593669633406535e-03
 	 3.74687219394429782415301133369212e-02 6.90350532054870402731694412068464e-01 2.72180746005686646782351090223528e-01 8.84020363336787337593669633406535e-03
 	 6.90350532054870402731694412068464e-01 2.72180746005686646782351090223528e-01 3.74687219394429504859544977080077e-02 8.84020363336787337593669633406535e-03
 	 9.90063515075153971478627568103548e-02 8.19687967813020335050566700374475e-02 8.19024851711182666491595227853395e-01 9.62913221988995898792396133103466e-03
 	 8.19687967813020335050566700374475e-02 8.19024851711182666491595227853395e-01 9.90063515075152444921968708513305e-02 9.62913221988995898792396133103466e-03
 	 8.19024851711182666491595227853395e-01 9.90063515075153971478627568103548e-02 8.19687967813019363605420153362502e-02 9.62913221988995898792396133103466e-03
 	 2.18932787550637497631100814032834e-01 7.39314412421514188800131250900449e-01 4.17528000278483135687679350667167e-02 9.80325260988398027028889458733829e-03
 	 7.39314412421514188800131250900449e-01 4.17528000278483135687679350667167e-02 2.18932787550637497631100814032834e-01 9.80325260988398027028889458733829e-03
 	 4.17528000278483135687679350667167e-02 2.18932787550637497631100814032834e-01 7.39314412421514188800131250900449e-01 9.80325260988398027028889458733829e-03
 	 3.85457856195294268353279676375678e-01 3.70114098745062680695383505735663e-02 5.77530733930199491332757588679669e-01 9.94281737582886245008229053610194e-03
 	 3.70114098745062680695383505735663e-02 5.77530733930199491332757588679669e-01 3.85457856195294268353279676375678e-01 9.94281737582886245008229053610194e-03
 	 5.77530733930199491332757588679669e-01 3.85457856195294268353279676375678e-01 3.70114098745062403139627349446528e-02 9.94281737582886245008229053610194e-03
 	 4.53931532973698426136621719706454e-01 5.07124120567042480089980927004945e-01 3.89443464592591492845485845464282e-02 1.07482407367000144110535941877060e-02
 	 5.07124120567042480089980927004945e-01 3.89443464592591492845485845464282e-02 4.53931532973698370625470488448627e-01 1.07482407367000144110535941877060e-02
 	 3.89443464592591492845485845464282e-02 4.53931532973698426136621719706454e-01 5.07124120567042480089980927004945e-01 1.07482407367000144110535941877060e-02
 	 3.31414067992492400449577871768270e-01 6.27872100319265880052910233644070e-01 4.07138316882417194975118945876602e-02 1.07752726636044619812349054654987e-02
 	 6.27872100319265880052910233644070e-01 4.07138316882417194975118945876602e-02 3.31414067992492400449577871768270e-01 1.07752726636044619812349054654987e-02
 	 4.07138316882417194975118945876602e-02 3.31414067992492400449577871768270e-01 6.27872100319265880052910233644070e-01 1.07752726636044619812349054654987e-02
 	 1.75418478700220814570087668471388e-01 1.09788999485381039145082127106434e-01 7.14792521814398118529254588793265e-01 1.17830977528251903962974722617219e-02
 	 1.09788999485381039145082127106434e-01 7.14792521814398118529254588793265e-01 1.75418478700220870081238899729215e-01 1.17830977528251903962974722617219e-02
 	 7.14792521814398118529254588793265e-01 1.75418478700220814570087668471388e-01 1.09788999485381066900657742735348e-01 1.17830977528251903962974722617219e-02
 	 2.99424606258227421484718888677889e-01 8.95177416056505692587563771667192e-02 6.11057652136121953745373502897564e-01 1.21567676541047674926288024721543e-02
 	 8.95177416056505692587563771667192e-02 6.11057652136121953745373502897564e-01 2.99424606258227532507021351193544e-01 1.21567676541047674926288024721543e-02
 	 6.11057652136121953745373502897564e-01 2.99424606258227421484718888677889e-01 8.95177416056506247699076084245462e-02 1.21567676541047674926288024721543e-02
 	 1.64438284214270008520841770405241e-01 7.35720413124063998822066423599608e-01 9.98413026616660204126674216240644e-02 1.29457052361328234235671175156313e-02
 	 7.35720413124063998822066423599608e-01 9.98413026616660204126674216240644e-02 1.64438284214269980765266154776327e-01 1.29457052361328234235671175156313e-02
 	 9.98413026616660204126674216240644e-02 1.64438284214270008520841770405241e-01 7.35720413124063998822066423599608e-01 1.29457052361328234235671175156313e-02
 	 3.69724758939286801684431793546537e-01 4.63127807453672823978507722131326e-01 1.67147433607040374337060484322137e-01 1.35548997474791085848799809809861e-02
 	 4.63127807453672823978507722131326e-01 1.67147433607040374337060484322137e-01 3.69724758939286746173280562288710e-01 1.35548997474791085848799809809861e-02
 	 1.67147433607040374337060484322137e-01 3.69724758939286801684431793546537e-01 4.63127807453672768467356490873499e-01 1.35548997474791085848799809809861e-02
 	 4.06369061777283435699104074956267e-01 9.07167861113062495848069488602050e-02 5.02914152111410217571574321482331e-01 1.46758324930695502330557644654618e-02
 	 9.07167861113062495848069488602050e-02 5.02914152111410217571574321482331e-01 4.06369061777283491210255306214094e-01 1.46758324930695502330557644654618e-02
 	 5.02914152111410217571574321482331e-01 4.06369061777283435699104074956267e-01 9.07167861113063467293216035614023e-02 1.46758324930695502330557644654618e-02
 	 2.67420664299267585306552064139396e-01 6.32902725169850244313352050085086e-01 9.96766105308821703800958857755177e-02 1.54204544583998980067374162672422e-02
 	 6.32902725169850244313352050085086e-01 9.96766105308821703800958857755177e-02 2.67420664299267585306552064139396e-01 1.54204544583998980067374162672422e-02
 	 9.96766105308821703800958857755177e-02 2.67420664299267585306552064139396e-01 6.32902725169850244313352050085086e-01 1.54204544583998980067374162672422e-02
 	 3.83204107909751179317225933118607e-01 5.21623720050690464766773857263615e-01 9.51721720395583004048489783599507e-02 1.55159977941551948527054705095907e-02
 	 5.21623720050690464766773857263615e-01 9.51721720395583004048489783599507e-02 3.83204107909751234828377164376434e-01 1.55159977941551948527054705095907e-02
 	 9.51721720395583004048489783599507e-02 3.83204107909751179317225933118607e-01 5.21623720050690575789076319779269e-01 1.55159977941551948527054705095907e-02
 	 1.78724492320244954246533097830252e-01 1.92031135757774140682130337154376e-01 6.29244371921980905071336565015372e-01 1.56625893060099094511450346089987e-02
 	 1.92031135757774140682130337154376e-01 6.29244371921980905071336565015372e-01 1.78724492320244898735381866572425e-01 1.56625893060099094511450346089987e-02
 	 6.29244371921980905071336565015372e-01 1.78724492320244954246533097830252e-01 1.92031135757774140682130337154376e-01 1.56625893060099094511450346089987e-02
 	 2.82829128593355183518553985777544e-01 5.37642437490313485071169452567119e-01 1.79528433916331331410276561655337e-01 1.59275375108478001173750726593425e-02
 	 5.37642437490313485071169452567119e-01 1.79528433916331331410276561655337e-01 2.82829128593355183518553985777544e-01 1.59275375108478001173750726593425e-02
 	 1.79528433916331331410276561655337e-01 2.82829128593355183518553985777544e-01 5.37642437490313485071169452567119e-01 1.59275375108478001173750726593425e-02
 	 2.70606479254229259634456639105338e-01 1.61339950052654446110977914941031e-01 5.68053570693116238743414214695804e-01 1.61035330300796433733445667257911e-02
 	 1.61339950052654446110977914941031e-01 5.68053570693116238743414214695804e-01 2.70606479254229315145607870363165e-01 1.61035330300796433733445667257911e-02
 	 5.68053570693116238743414214695804e-01 2.70606479254229259634456639105338e-01 1.61339950052654501622129146198859e-01 1.61035330300796433733445667257911e-02
 	 3.70874346744923255592141231318237e-01 1.64548573538222442369516329563339e-01 4.64577079716854302038342439118423e-01 1.66580804257312579952365894087052e-02
 	 1.64548573538222442369516329563339e-01 4.64577079716854302038342439118423e-01 3.70874346744923311103292462576064e-01 1.66580804257312579952365894087052e-02
 	 4.64577079716854302038342439118423e-01 3.70874346744923255592141231318237e-01 1.64548573538222497880667560821166e-01 1.66580804257312579952365894087052e-02
 	 3.63840811889283188840238381089875e-01 3.83367191770047632815021643182263e-01 2.52791996340669178344739975727862e-01 1.93530794923629304027379305352952e-02
 	 3.83367191770047632815021643182263e-01 2.52791996340669178344739975727862e-01 3.63840811889283188840238381089875e-01 1.93530794923629304027379305352952e-02
 	 2.52791996340669178344739975727862e-01 3.63840811889283188840238381089875e-01 3.83367191770047632815021643182263e-01 1.93530794923629304027379305352952e-02
 	 2.75735321862771332046548877769965e-01 2.52760449512089535595293909864267e-01 4.71504228625139076847005981107941e-01 2.04593586377354516581128507368703e-02
 	 2.52760449512089535595293909864267e-01 4.71504228625139076847005981107941e-01 2.75735321862771387557700109027792e-01 2.04593586377354516581128507368703e-02
 	 4.71504228625139076847005981107941e-01 2.75735321862771332046548877769965e-01 2.52760449512089591106445141122094e-01 2.04593586377354516581128507368703e-02
 	 ];
 
 	 case  25 

 	 % ALG. DEG.:   25
 	 % PTS CARD.:  117
 	 % NEG. W.  :    0
 	 % OUT PTS. :    0
 	 % M.E.INF. : 5.4e-16  

 	 xyw_bar=[
 	 1.31881771474051147363670821732740e-02 3.07956384011914569703849409165741e-03 9.83732259012475740433956161723472e-01 5.26947810971040804181586558740946e-04
 	 3.07956384011914569703849409165741e-03 9.83732259012475740433956161723472e-01 1.31881771474051667780713614774868e-02 5.26947810971040804181586558740946e-04
 	 9.83732259012475740433956161723472e-01 1.31881771474051147363670821732740e-02 3.07956384011914482967675610325387e-03 5.26947810971040804181586558740946e-04
 	 2.72596213613051892646343965509459e-03 1.71043047595741092015142470472711e-02 9.80169733104295337611233662755694e-01 5.38378571248994018325073618314036e-04
 	 1.71043047595741092015142470472711e-02 9.80169733104295337611233662755694e-01 2.72596213613052196222952261450700e-03 5.38378571248994018325073618314036e-04
 	 9.80169733104295337611233662755694e-01 2.72596213613051892646343965509459e-03 1.71043047595741438959837665834129e-02 5.38378571248994018325073618314036e-04
 	 6.29840020924785359834885412055883e-02 9.31371984168810507220825911645079e-01 5.64401373871092904010993152041920e-03 1.71255729388201764773480562098484e-03
 	 9.31371984168810507220825911645079e-01 5.64401373871092904010993152041920e-03 6.29840020924785637390641568345018e-02 1.71255729388201764773480562098484e-03
 	 5.64401373871092904010993152041920e-03 6.29840020924785359834885412055883e-02 9.31371984168810507220825911645079e-01 1.71255729388201764773480562098484e-03
 	 5.20238006731535140847455522816745e-02 7.43791496086179931140902610309240e-03 9.40538284365984744717081866838271e-01 1.91233286697058395969817112813871e-03
 	 7.43791496086179931140902610309240e-03 9.40538284365984744717081866838271e-01 5.20238006731534863291699366527610e-02 1.91233286697058395969817112813871e-03
 	 9.40538284365984744717081866838271e-01 5.20238006731535140847455522816745e-02 7.43791496086174119817258088005474e-03 1.91233286697058395969817112813871e-03
 	 2.20200206141671100956713758023398e-01 7.76414334424490371056037929520244e-01 3.38545943383850023167269682744518e-03 1.98974451400670687695182436982577e-03
 	 7.76414334424490371056037929520244e-01 3.38545943383850023167269682744518e-03 2.20200206141671128712289373652311e-01 1.98974451400670687695182436982577e-03
 	 3.38545943383850023167269682744518e-03 2.20200206141671100956713758023398e-01 7.76414334424490371056037929520244e-01 1.98974451400670687695182436982577e-03
 	 3.04395561568152825393518412511185e-02 9.44856451527789409361446359980619e-01 2.47039923153953289158835104899481e-02 2.70300175161245935170906129485502e-03
 	 9.44856451527789409361446359980619e-01 2.47039923153953289158835104899481e-02 3.04395561568152617226701295294333e-02 2.70300175161245935170906129485502e-03
 	 2.47039923153953289158835104899481e-02 3.04395561568152825393518412511185e-02 9.44856451527789409361446359980619e-01 2.70300175161245935170906129485502e-03
 	 1.13993174377526096385793152876431e-01 7.47102931577944221613885034116720e-03 8.78535796306694427570960215234663e-01 2.82376214206483521634849509496235e-03
 	 7.47102931577944221613885034116720e-03 8.78535796306694427570960215234663e-01 1.13993174377526096385793152876431e-01 2.82376214206483521634849509496235e-03
 	 8.78535796306694427570960215234663e-01 1.13993174377526096385793152876431e-01 7.47102931577947604324663188890554e-03 2.82376214206483521634849509496235e-03
 	 3.42295210188022336339486173528712e-01 6.52613563397589468983994720474584e-01 5.09122641438819467651910599670373e-03 2.91514917247315307261001926519839e-03
 	 6.52613563397589468983994720474584e-01 5.09122641438819467651910599670373e-03 3.42295210188022336339486173528712e-01 2.91514917247315307261001926519839e-03
 	 5.09122641438819467651910599670373e-03 3.42295210188022336339486173528712e-01 6.52613563397589468983994720474584e-01 2.91514917247315307261001926519839e-03
 	 1.30473991950559503605333588893700e-01 8.61524966076299802963944785005879e-01 8.00104197314066567514601047150791e-03 3.14622878230012923556069281971759e-03
 	 8.61524966076299802963944785005879e-01 8.00104197314066567514601047150791e-03 1.30473991950559531360909204522613e-01 3.14622878230012923556069281971759e-03
 	 8.00104197314066567514601047150791e-03 1.30473991950559503605333588893700e-01 8.61524966076299802963944785005879e-01 3.14622878230012923556069281971759e-03
 	 2.00179308343867129460491582904069e-01 6.50781868302953165478230701523898e-03 7.93312872973103377916004319558851e-01 3.25439774886752226576436264338099e-03
 	 6.50781868302953165478230701523898e-03 7.93312872973103377916004319558851e-01 2.00179308343867101704915967275156e-01 3.25439774886752226576436264338099e-03
 	 7.93312872973103377916004319558851e-01 2.00179308343867129460491582904069e-01 6.50781868302949262350409753707936e-03 3.25439774886752226576436264338099e-03
 	 3.07372363215433874739090924776974e-01 6.35240709459219433929977327579763e-03 6.86275229689973897961863258387893e-01 3.72306578149784973483238381675164e-03
 	 6.35240709459219433929977327579763e-03 6.86275229689973897961863258387893e-01 3.07372363215433930250242156034801e-01 3.72306578149784973483238381675164e-03
 	 6.86275229689973897961863258387893e-01 3.07372363215433874739090924776974e-01 6.35240709459222729904581683513243e-03 3.72306578149784973483238381675164e-03
 	 2.76217326622710424022244524167036e-01 7.06370557783410402130641614348860e-01 1.74121155938792293582650927419309e-02 4.30631473928152954527837081855068e-03
 	 7.06370557783410402130641614348860e-01 1.74121155938792293582650927419309e-02 2.76217326622710368511093292909209e-01 4.30631473928152954527837081855068e-03
 	 1.74121155938792293582650927419309e-02 2.76217326622710424022244524167036e-01 7.06370557783410291108339151833206e-01 4.30631473928152954527837081855068e-03
 	 4.22935483077426899178874464269029e-01 7.56275469533758924745026064329068e-03 5.69501762227235475144482279574731e-01 4.52513415896231303153784963910766e-03
 	 7.56275469533758924745026064329068e-03 5.69501762227235475144482279574731e-01 4.22935483077426899178874464269029e-01 4.52513415896231303153784963910766e-03
 	 5.69501762227235475144482279574731e-01 4.22935483077426899178874464269029e-01 7.56275469533762567664325615623966e-03 4.52513415896231303153784963910766e-03
 	 4.53227003366671432615930825704709e-01 5.38998045885384025055486745259259e-01 7.77495074794454232858242903603241e-03 4.56249922131780807216250295255122e-03
 	 5.38998045885384025055486745259259e-01 7.77495074794454232858242903603241e-03 4.53227003366671432615930825704709e-01 4.56249922131780807216250295255122e-03
 	 7.77495074794454232858242903603241e-03 4.53227003366671432615930825704709e-01 5.38998045885384025055486745259259e-01 4.56249922131780807216250295255122e-03
 	 1.90195202693624082490586602034455e-01 7.84036511910398581370884585339809e-01 2.57682853959773083829531969968230e-02 4.68027662060985401304424868840215e-03
 	 7.84036511910398581370884585339809e-01 2.57682853959773083829531969968230e-02 1.90195202693624110246162217663368e-01 4.68027662060985401304424868840215e-03
 	 2.57682853959773083829531969968230e-02 1.90195202693624082490586602034455e-01 7.84036511910398581370884585339809e-01 4.68027662060985401304424868840215e-03
 	 3.61718299806125034856485456202790e-02 8.44564005292429142990329182794085e-02 8.79371769490144616909788055636454e-01 5.64055598154384295328522824775064e-03
 	 8.44564005292429142990329182794085e-02 8.79371769490144616909788055636454e-01 3.61718299806124132800277948263101e-02 5.64055598154384295328522824775064e-03
 	 8.79371769490144616909788055636454e-01 3.61718299806125034856485456202790e-02 8.44564005292428865434573026504950e-02 5.64055598154384295328522824775064e-03
 	 7.60794388659036535660362687849556e-02 4.05076326129555214228261661446595e-02 8.83412928521140838888925372884842e-01 5.77842305077480751507312817238926e-03
 	 4.05076326129555214228261661446595e-02 8.83412928521140838888925372884842e-01 7.60794388659036258104606531560421e-02 5.77842305077480751507312817238926e-03
 	 8.83412928521140838888925372884842e-01 7.60794388659036535660362687849556e-02 4.05076326129555075450383583302028e-02 5.77842305077480751507312817238926e-03
 	 1.49945977487857845522611910382693e-01 3.58350589330340557614462682067824e-02 8.14218963579108057082578397967154e-01 6.57914135071524693032518626978344e-03
 	 3.58350589330340557614462682067824e-02 8.14218963579108057082578397967154e-01 1.49945977487857873278187526011607e-01 6.57914135071524693032518626978344e-03
 	 8.14218963579108057082578397967154e-01 1.49945977487857845522611910382693e-01 3.58350589330340973948096916501527e-02 6.57914135071524693032518626978344e-03
 	 1.50955341179002944240750139215379e-01 7.96913702682935043419831799837993e-01 5.21309561380619568282668296888005e-02 6.72965316584276176947732039934635e-03
 	 7.96913702682935043419831799837993e-01 5.21309561380619568282668296888005e-02 1.50955341179002999751901370473206e-01 6.72965316584276176947732039934635e-03
 	 5.21309561380619568282668296888005e-02 1.50955341179002944240750139215379e-01 7.96913702682935154442134262353648e-01 6.72965316584276176947732039934635e-03
 	 2.44783310976574935979854785728094e-01 3.26897799308286454711414137364045e-02 7.22526909092596425487897704442730e-01 7.57554243586837025570268622232106e-03
 	 3.26897799308286454711414137364045e-02 7.22526909092596425487897704442730e-01 2.44783310976574908224279170099180e-01 7.57554243586837025570268622232106e-03
 	 7.22526909092596425487897704442730e-01 2.44783310976574935979854785728094e-01 3.26897799308286385322475098291761e-02 7.57554243586837025570268622232106e-03
 	 1.70674433667943242776132706239878e-01 7.87558693381886776307965192245319e-02 7.50569696993868107348646390164504e-01 8.14583504633362848190714089469111e-03
 	 7.87558693381886776307965192245319e-02 7.50569696993868107348646390164504e-01 1.70674433667943215020557090610964e-01 8.14583504633362848190714089469111e-03
 	 7.50569696993868107348646390164504e-01 1.70674433667943242776132706239878e-01 7.87558693381886498752209035956184e-02 8.14583504633362848190714089469111e-03
 	 3.71942611691943147977923445068882e-01 5.93428569491770030452926221187226e-01 3.46288188162868770803015650017187e-02 9.08183523962656329142628663930736e-03
 	 5.93428569491770030452926221187226e-01 3.46288188162868770803015650017187e-02 3.71942611691943092466772213811055e-01 9.08183523962656329142628663930736e-03
 	 3.46288188162868770803015650017187e-02 3.71942611691943147977923445068882e-01 5.93428569491770030452926221187226e-01 9.08183523962656329142628663930736e-03
 	 3.51792010432223745119983959739329e-01 3.63528731441380772992921777131414e-02 6.11855116423638212275193382083671e-01 9.40089530430836088636947778240938e-03
 	 3.63528731441380772992921777131414e-02 6.11855116423638212275193382083671e-01 3.51792010432223745119983959739329e-01 9.40089530430836088636947778240938e-03
 	 6.11855116423638212275193382083671e-01 3.51792010432223745119983959739329e-01 3.63528731441380426048226581769995e-02 9.40089530430836088636947778240938e-03
 	 9.84400611472149716929536111820198e-02 9.69591386055578818847777711198432e-02 8.04600800247227132544480809883680e-01 9.52400454407075283680406130315532e-03
 	 9.69591386055578818847777711198432e-02 8.04600800247227132544480809883680e-01 9.84400611472149300595901877386495e-02 9.52400454407075283680406130315532e-03
 	 8.04600800247227132544480809883680e-01 9.84400611472149716929536111820198e-02 9.69591386055578957625655789343000e-02 9.52400454407075283680406130315532e-03
 	 2.60865026298107871571829718959634e-01 6.84089653184228985161041691753780e-01 5.50453205176631987782798205444124e-02 1.01072081452672178636387201322577e-02
 	 6.84089653184228985161041691753780e-01 5.50453205176631987782798205444124e-02 2.60865026298107816060678487701807e-01 1.01072081452672178636387201322577e-02
 	 5.50453205176631987782798205444124e-02 2.60865026298107871571829718959634e-01 6.84089653184228874138739229238126e-01 1.01072081452672178636387201322577e-02
 	 3.66582144327067316336155045064515e-01 5.50781902551756963326567984040594e-01 8.26359531211756648261257396370638e-02 1.08591268077148413867538678800884e-02
 	 5.50781902551756963326567984040594e-01 8.26359531211756648261257396370638e-02 3.66582144327067371847306276322342e-01 1.08591268077148413867538678800884e-02
 	 8.26359531211756648261257396370638e-02 3.66582144327067316336155045064515e-01 5.50781902551757074348870446556248e-01 1.08591268077148413867538678800884e-02
 	 4.68735987743839010999380434441264e-01 4.11368221710290485337857546710438e-02 4.90127190085131891894576483537094e-01 1.11689841862876882566668612639660e-02
 	 4.11368221710290485337857546710438e-02 4.90127190085131891894576483537094e-01 4.68735987743839010999380434441264e-01 1.11689841862876882566668612639660e-02
 	 4.90127190085131891894576483537094e-01 4.68735987743839010999380434441264e-01 4.11368221710291526171943132794695e-02 1.11689841862876882566668612639660e-02
 	 2.57617931613541240043474545018398e-01 8.24865172853232742067319804846193e-02 6.59895551101135513505369090125896e-01 1.17722968652321513527425622669398e-02
 	 8.24865172853232742067319804846193e-02 6.59895551101135513505369090125896e-01 2.57617931613541184532323313760571e-01 1.17722968652321513527425622669398e-02
 	 6.59895551101135513505369090125896e-01 2.57617931613541240043474545018398e-01 8.24865172853232464511563648557058e-02 1.17722968652321513527425622669398e-02
 	 1.84517952412833835529326620417123e-01 7.12575925803732079799601706326939e-01 1.02906121783434056915496057627024e-01 1.21424040084771537456598267112895e-02
 	 7.12575925803732079799601706326939e-01 1.02906121783434056915496057627024e-01 1.84517952412833863284902236046037e-01 1.21424040084771537456598267112895e-02
 	 1.02906121783434056915496057627024e-01 1.84517952412833835529326620417123e-01 7.12575925803732079799601706326939e-01 1.21424040084771537456598267112895e-02
 	 2.81209118651227718554963530550594e-01 6.05257308197521104453642237785971e-01 1.13533573151251121480243000405608e-01 1.25526989266996867600223453109720e-02
 	 6.05257308197521104453642237785971e-01 1.13533573151251121480243000405608e-01 2.81209118651227774066114761808421e-01 1.25526989266996867600223453109720e-02
 	 1.13533573151251121480243000405608e-01 2.81209118651227718554963530550594e-01 6.05257308197521215475944700301625e-01 1.25526989266996867600223453109720e-02
 	 1.73591491675633341484896732254128e-01 1.42209318699190873092064180127636e-01 6.84199189625175785423039087618235e-01 1.27814412760983882322296523170735e-02
 	 1.42209318699190873092064180127636e-01 6.84199189625175785423039087618235e-01 1.73591491675633369240472347883042e-01 1.27814412760983882322296523170735e-02
 	 6.84199189625175785423039087618235e-01 1.73591491675633341484896732254128e-01 1.42209318699190873092064180127636e-01 1.27814412760983882322296523170735e-02
 	 4.42362027995821582848634534457233e-01 4.54988327657975588635963504202664e-01 1.02649644346202828515401961340103e-01 1.32468413876246516452939161467839e-02
 	 4.54988327657975588635963504202664e-01 1.02649644346202828515401961340103e-01 4.42362027995821582848634534457233e-01 1.32468413876246516452939161467839e-02
 	 1.02649644346202828515401961340103e-01 4.42362027995821582848634534457233e-01 4.54988327657975588635963504202664e-01 1.32468413876246516452939161467839e-02
 	 3.59368226555797587451479557785206e-01 9.07953156997939186956259050020890e-02 5.49836457744408479975106729398249e-01 1.38282415635121516683803122305108e-02
 	 9.07953156997939186956259050020890e-02 5.49836457744408479975106729398249e-01 3.59368226555797587451479557785206e-01 1.38282415635121516683803122305108e-02
 	 5.49836457744408479975106729398249e-01 3.59368226555797587451479557785206e-01 9.07953156997939325734137128165457e-02 1.38282415635121516683803122305108e-02
 	 1.82134268926742365346527208203042e-01 2.28667042140729831478651590259688e-01 5.89198688932527803174821201537270e-01 1.59282121621503146291498609343762e-02
 	 2.28667042140729831478651590259688e-01 5.89198688932527803174821201537270e-01 1.82134268926742337590951592574129e-01 1.59282121621503146291498609343762e-02
 	 5.89198688932527803174821201537270e-01 1.82134268926742365346527208203042e-01 2.28667042140729831478651590259688e-01 1.59282121621503146291498609343762e-02
 	 2.67343042736282066051956007868284e-01 1.56184413818651390837288772672764e-01 5.76472543445066598621906450716779e-01 1.67459571385038495650032785988515e-02
 	 1.56184413818651390837288772672764e-01 5.76472543445066598621906450716779e-01 2.67343042736281955029653545352630e-01 1.67459571385038495650032785988515e-02
 	 5.76472543445066598621906450716779e-01 2.67343042736282066051956007868284e-01 1.56184413818651335326137541414937e-01 1.67459571385038495650032785988515e-02
 	 3.65290320906562715475018876531976e-01 1.73792802318537387140651162553695e-01 4.60916876774899841873178729656502e-01 1.87746105537932306273063431945047e-02
 	 1.73792802318537387140651162553695e-01 4.60916876774899841873178729656502e-01 3.65290320906562826497321339047630e-01 1.87746105537932306273063431945047e-02
 	 4.60916876774899841873178729656502e-01 3.65290320906562715475018876531976e-01 1.73792802318537387140651162553695e-01 1.87746105537932306273063431945047e-02
 	 3.45064826601384877502454173736623e-01 4.82299125389173410916043849283597e-01 1.72636048009441656070350745721953e-01 1.97380050801869745547012513497975e-02
 	 4.82299125389173410916043849283597e-01 1.72636048009441656070350745721953e-01 3.45064826601384933013605404994451e-01 1.97380050801869745547012513497975e-02
 	 1.72636048009441656070350745721953e-01 3.45064826601384877502454173736623e-01 4.82299125389173466427195080541424e-01 1.97380050801869745547012513497975e-02
 	 2.65562227862875310790968796936795e-01 2.56802629703844209618779359516338e-01 4.77635142433280479590251843546866e-01 2.02862843586862802436332486877291e-02
 	 2.56802629703844209618779359516338e-01 4.77635142433280479590251843546866e-01 2.65562227862875366302120028194622e-01 2.02862843586862802436332486877291e-02
 	 4.77635142433280479590251843546866e-01 2.65562227862875310790968796936795e-01 2.56802629703844154107628128258511e-01 2.02862843586862802436332486877291e-02
 	 3.62956385501777445412585620942991e-01 3.71550947373690598762863146475866e-01 2.65492667124531955824551232581143e-01 2.16253435779476220479633497006944e-02
 	 3.71550947373690598762863146475866e-01 2.65492667124531955824551232581143e-01 3.62956385501777389901434389685164e-01 2.16253435779476220479633497006944e-02
 	 2.65492667124531955824551232581143e-01 3.62956385501777445412585620942991e-01 3.71550947373690654274014377733693e-01 2.16253435779476220479633497006944e-02
 	 ];
 

 	 otherwise
 	 xyw=[];
 	 end
 

 
 
 	 pointset_stats=[
 	  10   24 2.08e-15    0    0
 	  11   27 9.92e-16    0    0
 	  12   31 2.21e-15    0    3
 	  13   36 3.05e-16    0    0
 	  14   40 9.51e-16    0    3
 	  15   46 5.00e-16    0    0
 	  16   51 1.78e-15    0    6
 	  17   57 5.83e-16    0    0
 	  18   64 4.86e-16    1    0
 	  19   70 4.16e-16    0    0
 	  20   78 4.89e-16    0    0
 	  21   85 4.64e-16    0    0
 	  22   93 7.56e-16    0    0
 	  23  100 6.66e-16    0    0
 	  24  109 7.81e-16    0    0
 	  25  117 5.41e-16    0    0
 	 ];
 
 
