
function [xyw,pointset_stats]=set_papanicolopulos_A_IV_C_standard(deg)

%--------------------------------------------------------------------
% CUBATURE RULE ON THE SIMPLEX [0 0; 0 1; 1 0] WITH ADE EQUAL TO deg.
% THE CUBATURE RULE IS DESCRIBED IN STANDARD FORM.
%--------------------------------------------------------------------
% INPUT:
% deg: algebraic degree of precision (ADE) of the rule.
%--------------------------------------------------------------------
% OUTPUT:
% xyw: MATRIX OF NODES xyw(:,1:2) AND WEIGHTS xyw(:,3).
%      REMARK: The sum of the weights makes 1/2.
%
% pointset_stats:
%    statistics of the pointset stored in a matrix. Each rows contains
%    degree, cardinality of the full pointset, moment error w.r.t. orthn.
%    Dubiner basis,number of points with negative weights, number of
%    points outside the domain.
%--------------------------------------------------------------------
% SOURCE:
% S-A. Papanicoloupolos
% "New fully symmetric and rotationally symmetric cubature rules on the
% triangle using minimal orthonormal bases".
% JCAM 294 (2016), 39-48.
% ** Using file rotationalsymmetry.txt from archives.
%--------------------------------------------------------------------
% REMARK:
% Rules in the original set not appropriate for cubature when deg <= 16.
%--------------------------------------------------------------------
% | DEG | CARD | W- | PT.O|   M.E.   |Q.|
%--------------------------------------------------------------------
% |  10 |  24 |   0 |   0 | 2.08e-15 |PI|
% |  11 |  27 |   0 |   0 | 9.92e-16 |PI|
% |  12 |  31 |   0 |   3 | 2.21e-15 |P0|
% |  13 |  36 |   0 |   0 | 3.05e-16 |PI|
% |  14 |  40 |   0 |   3 | 9.51e-16 |P0|
% |  15 |  46 |   0 |   0 | 5.00e-16 |PI|
% |  16 |  51 |   0 |   6 | 1.78e-15 |P0|
% |  17 |  57 |   0 |   0 | 5.83e-16 |PI|
% |  18 |  64 |   1 |   0 | 4.86e-16 |NI|
% |  19 |  70 |   0 |   0 | 4.16e-16 |PI|
% |  20 |  78 |   0 |   0 | 4.89e-16 |PI|
% |  21 |  85 |   0 |   0 | 4.64e-16 |PI|
% |  22 |  93 |   0 |   0 | 7.56e-16 |PI|
% |  23 | 100 |   0 |   0 | 6.66e-16 |PI|
% |  24 | 109 |   0 |   0 | 7.81e-16 |PI|
% |  25 | 117 |   0 |   0 | 5.41e-16 |PI|
%--------------------------------------------------------------------
% LEGEND:
%--------------------------------------------------------------------
% DEG  : algebraic degree of precision
% CARD : cardinality of the pointset
% W-   : negative weights
% PT.O : points outside the simplex [0 0; 1 0; 0 1].
% M.E. : maximum moment error in inf norm.
% Q.: quality of the rule.
%        PI: pos. weights, points in simplex
%        NI: some neg. weights, points in simplex
%        P0: pos. weights, some points out of the simplex
%        N0: some neg. weights, some points out of the simplex
%--------------------------------------------------------------------


switch deg
    
    case  10
        
        % ALG. DEG.:   10
        % PTS CARD.:   24
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 2.1e-15
        
        xyw=[
            1.45834576989294056881263372815738e-01 1.13985726592417714597704758716645e-02 7.35557170162471057189845780044379e-03
            1.13985706737416217504410553829075e-02 8.42766851850608622598315378127154e-01 7.35557125241108940988388198434222e-03
            8.42766841218143714087318585370667e-01 1.45834585819399425998810215787671e-01 7.35557198827177950023470387463931e-03
            4.80639357548164503075582842939184e-01 3.96436775306336502966431112326973e-03 7.46429581089971035806618360197717e-03
            3.96437233272861535432474155982163e-03 5.15396271752390666698318000271684e-01 7.46429685580188729404627068220179e-03
            5.15396271752390666698318000271684e-01 4.80639356407453255393136259954190e-01 7.46429708417026792355297715175766e-03
            2.83843627688640069706149660078154e-03 2.35341709211809968493867017969023e-01 7.47366178939868567449034131300323e-03
            2.35341708166612734176581511746917e-01 7.61819851839867112630599876865745e-01 7.47366271793433074865919962803673e-03
            7.61819857564093094737245337455533e-01 2.83843669030111114826020468626666e-03 7.47366189523436617175855545269769e-03
            2.73824571185072064560817040046459e-02 4.33992887961679035813133964438748e-02 7.67850037898878522851786598835133e-03
            4.33992891185142615206871141708689e-02 9.29218253512416048067734664073214e-01 7.67850048912587391375383916169994e-03
            9.29218252541391342980148237984395e-01 2.73824594333478416041671721359307e-02 7.67850083986368423666357685419825e-03
            2.90369914929698991201689750596415e-01 5.94849894428558720327337994149275e-02 2.57202510447084101152803725653939e-02
            5.94849886320373416603857208428963e-02 6.50145097261727644699647044035373e-01 2.57202503546237644382888021254985e-02
            6.50145090521004664907422920805402e-01 2.90369916954274109865963282572920e-01 2.57202510445128339211517953799557e-02
            1.02579103608621838295000827656622e-01 1.45523272507306150602701677598816e-01 2.75936962858097448170191512417659e-02
            1.45523268651140014195988214851241e-01 7.51897624119613472437606560561107e-01 2.75936959210027685229782434817025e-02
            7.51897620213141881784224551665829e-01 1.02579105494409267285149667259248e-01 2.75936965602863529256616459406359e-02
            8.91387185038082507926304742795764e-02 3.68934748340993656778863396539236e-01 3.58589896358298548806331496052735e-02
            3.68934745933899244096920710944687e-01 5.41926532339293443030214803002309e-01 3.58589903617937480473365496891347e-02
            5.41926530667821038100839814433130e-01 8.91387164194035602760024517010606e-02 3.58589901805992719574511795599392e-02
            2.89306187176388229786994088499341e-01 2.27196143161295271983846078001079e-01 4.75216984060985586646452816239616e-02
            2.27196140848565653058699354005512e-01 4.83497668078035791605628901379532e-01 4.75216985441686765034852157896239e-02
            4.83497664803317683812622362893308e-01 2.89306187540847858929993208221276e-01 4.75216988568408355009253796197299e-02
            ];
        
    case  11
        
        % ALG. DEG.:   11
        % PTS CARD.:   27
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 9.9e-16
        
        xyw=[
            3.17327240959096867900512961568893e-02 3.38153709534936494640611215345416e-02 6.90364016068318979424400794187022e-03
            3.38153709571890753737832824299403e-02 9.34451904826319457164629511680687e-01 6.90364018872116277925066185616743e-03
            9.34451904771643859781704577471828e-01 3.17327242005391030321703738081851e-02 6.90364019898256821988757891972455e-03
            1.59837359530232481219513829273637e-01 2.47248304899685064639491827165330e-02 1.08516175752374158941515602805339e-02
            2.47248303570562662223597527599850e-02 8.15437809424427229032517061568797e-01 1.08516175719021723544699398189550e-02
            8.15437809808309377324064826098038e-01 1.59837360119939320135529214894632e-01 1.08516174533720300560135640921544e-02
            2.95071648130748806493350855362223e-02 1.69403881037778814011574013420613e-01 1.35170641807693010794100274551965e-02
            1.69403881176198312097369580442319e-01 8.01088954177139322965217616001610e-01 1.35170641232079061927473517812359e-02
            8.01088953864914521396656255092239e-01 2.95071646435425549714892667907407e-02 1.35170641456389181017661016426246e-02
            2.28225758126171518469504917447921e-02 3.82904117992176939999637852451997e-01 1.36656203286281661968271095020100e-02
            3.82904118304647045345490141698974e-01 5.94273305888574410182911833544495e-01 1.36656203271459889686534339148238e-02
            5.94273305139020657783532897155965e-01 2.28225758029828758310930680863748e-02 1.36656203506252977436519202569798e-02
            3.60374165922343669965499657337205e-01 2.51549434748986507071055740425436e-02 1.41681421964768485682872878328453e-02
            2.51549437595472435980070713412715e-02 6.14470889118425733776973629574059e-01 1.41681423689446108621359243784354e-02
            6.14470889369991835060602625162574e-01 3.60374166818971664039139568558312e-01 1.41681423854682318752562508734627e-02
            1.52649491517360930137314767307544e-01 1.30991018319333046848740309542336e-01 2.42718337805244814553429222314662e-02
            1.30991018089064076290739535579633e-01 7.16359490961503864170367705810349e-01 2.42718336975122697674045468829718e-02
            7.16359491749854693232180125050945e-01 1.52649491098747813744296308868797e-01 2.42718336544174330404732842225712e-02
            3.48698928000996666298760828794912e-01 1.20982495295537784141259862735751e-01 2.64436491378070813451639509139568e-02
            1.20982496547677217946237249179831e-01 5.30318577259008328006473220739281e-01 2.64436491983485698342715153330573e-02
            5.30318576743719294341872227960266e-01 3.48698926655225394100057201285381e-01 2.64436491997190013802931929376427e-02
            1.21263845339625003783012857638823e-01 3.19556007053755031233066574714030e-01 2.75811267290054888678962896619851e-02
            3.19556006508081080763616910189739e-01 5.59180148330025272862542351504089e-01 2.75811266607418718865840645548815e-02
            5.59180149334335019339903283253079e-01 1.21263845232059297440940781598329e-01 2.75811266316970567225030919189521e-02
            2.79902470374426504129417025978910e-01 2.69674312574123831165451292690705e-01 2.92639726964975081802133871633487e-02
            2.69674314114379864992798729872447e-01 4.50423216214933308521040089544840e-01 2.92639726589621236574334517399620e-02
            4.50423217577915646625541512548807e-01 2.79902468937386739167294535945985e-01 2.92639723989634240042256863034709e-02
            ];
        
    case  12
        
        % ALG. DEG.:   12
        % PTS CARD.:   31
        % NEG. W.  :    0
        % OUT PTS. :    3
        % M.E.INF. : 2.2e-15
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 1.07047093033107738185849555634377e-02
            3.01131981576056410865316337321929e-02 -9.47156971046334839048341791567509e-02 4.18345485042730339127724226155891e-05
            -9.47156971046355516952175435108074e-02 1.06460249894703018469499511411414e+00 4.18345485042701743295424920976444e-05
            1.06460249894702507766908183839405e+00 3.01131981576053600613285254894436e-02 4.18345485042771674335550236012438e-05
            3.28665292260699182236294291215017e-01 8.04564374091387701781741270679049e-03 6.20681762656306434261521687290042e-03
            8.04564374091384232334789317064860e-03 6.63289063998386274612073520984268e-01 6.20681762656306867942390681491815e-03
            6.63289063998386385634375983499922e-01 3.28665292260700070414713991340250e-01 6.20681762656298454533532193977408e-03
            4.81249505497031090928139462903346e-02 9.23861895067511262880088906968012e-01 8.11997596059716580940701646795787e-03
            9.23861895067510707768576594389742e-01 2.80131543827856974160361858139368e-02 8.11997596059722305528172370259199e-03
            2.80131543827856280270971467416530e-02 4.81249505497032964429493517855008e-02 8.11997596059717621774787232880044e-03
            1.42850847044125095486322152282810e-01 3.10192404956991518716336031502578e-02 1.20970198255959202754405978907926e-02
            3.10192404956990790132476121243599e-02 8.26129912460175530597439319535624e-01 1.20970198255959150712701699603713e-02
            8.26129912460175419575136857019970e-01 1.42850847044125428553229539829772e-01 1.20970198255959220101640738675997e-02
            2.08191967793620313909030983268167e-01 7.64831204293204724997679022635566e-01 1.34836824084055926370462330510236e-02
            7.64831204293203836819259322510334e-01 2.69768279131749472155021862818103e-02 1.34836824084056030453870889118662e-02
            2.69768279131749680321838980034954e-02 2.08191967793620591464787139557302e-01 1.34836824084055995759401369582520e-02
            4.25825852108942526363932756794384e-01 5.50395667341567640384880633064313e-01 1.38872121734323495517715585378937e-02
            5.50395667341566752206460932939081e-01 2.37784805494897950872701386515473e-02 1.38872121734323252656428948625944e-02
            2.37784805494898505984213699093743e-02 4.25825852108942748408537681825692e-01 1.38872121734323530212185104915079e-02
            3.23868231681058882376333940555924e-01 7.98513424486459177931152453311370e-02 2.35083890285216999538331350549925e-02
            7.98513424486458761597518218877667e-02 5.96280425870295172074975198484026e-01 2.35083890285217034232800870086066e-02
            5.96280425870295394119580123515334e-01 3.23868231681059215443241328102886e-01 2.35083890285217138316209428694492e-02
            1.54176409947630321806144593210774e-01 7.07647950434089900717538057506317e-01 2.42967073213548150067975228694195e-02
            7.07647950434089789695235594990663e-01 1.38175639618279860743044196169649e-01 2.42967073213548254151383787302620e-02
            1.38175639618279888498619811798562e-01 1.54176409947630516095173902613169e-01 2.42967073213548219456914267766479e-02
            3.49383622277121108634645452184486e-01 5.24861097355139305520310699648689e-01 2.92526399480571887290469845765983e-02
            5.24861097355139305520310699648689e-01 1.25755280367739502578317001280084e-01 2.92526399480572060762817443446693e-02
            1.25755280367739613600619463795738e-01 3.49383622277121219656947914700140e-01 2.92526399480571748512591767621416e-02
            2.99315867902454901816611254616873e-01 2.29249619433458784456547618901823e-01 3.22041513911976648443236115326727e-02
            2.29249619433458812212123234530736e-01 4.71434512664086480260294820254785e-01 3.22041513911976579054297076254443e-02
            4.71434512664086646793748514028266e-01 2.99315867902454846305460023359046e-01 3.22041513911977134165809388832713e-02
            ];
        
    case  13
        
        % ALG. DEG.:   13
        % PTS CARD.:   36
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.1e-16
        
        xyw=[
            2.60167889975679161862487376311037e-02 2.40222268841206107459207430565584e-02 4.03998964180703364068580896173444e-03
            2.40222266056275869650082910311539e-02 9.49960984299580535328288988239365e-01 4.03998960652205731103325803132975e-03
            9.49960983828827321140408912469866e-01 2.60167891895750139075182971737377e-02 4.03998968727853520477300008906241e-03
            1.88836306686942824450969879990225e-02 1.22080867325125158595611196687969e-01 6.44601445440000451331252762088297e-03
            1.22080865615096983978205003040785e-01 8.59035503703662994823275766975712e-01 6.44601436374914963517435140261114e-03
            8.59035502492874303293035609385697e-01 1.88836297321810057947555350210678e-02 6.44601419077840860627492958201401e-03
            1.31639216755138704906258340088243e-01 2.28747090565116009686086329111276e-02 8.28012183702025485187991904467708e-03
            2.28747093163145995275975508320698e-02 8.45486073780362001883759148768149e-01 8.28012192673547101073872056531400e-03
            8.45486073324166143017066588072339e-01 1.31639217770324529199754692854185e-01 8.28012185413436532788189481379959e-03
            2.07315314538747512418392915378718e-02 2.80737807897761981745787807085435e-01 9.51156673209609636321371084477505e-03
            2.80737804477865227070765286043752e-01 6.98530664767438969064983211865183e-01 9.51156633244431420903808316325012e-03
            6.98530660028288119534067845961545e-01 2.07315315728960856966889281238764e-02 9.51156678397508704991825112529114e-03
            4.76642854053482611931258361437358e-01 5.04849500035025267052901654096786e-01 9.67982549828267160862971252299758e-03
            5.04849495463474129408609769598115e-01 1.85076462972688683683575305849445e-02 9.67982564452105238850254664839667e-03
            1.85076458585603301210120719133556e-02 4.76642857835611943873033169438713e-01 9.67982542765623114877637789277287e-03
            3.02387014801126063900937879225239e-01 1.99757515090916715205793252607691e-02 9.79189853858823851251447223376090e-03
            1.99757511746916083250624751599389e-02 6.77637233570464792009602206235286e-01 9.79189840044913592520536838037515e-03
            6.77637231251634108986081628245302e-01 3.02387017597124940770925149990944e-01 9.79189846461464910576388831486838e-03
            1.01539777415501644708406558947900e-01 1.19028238394272706246468374047254e-01 1.51407481012283094690129559012348e-02
            1.19028239511687658480632023838552e-01 7.79431983178413778645676757150795e-01 1.51407482760150848000169077067767e-02
            7.79431987279475047003529653011356e-01 1.01539774383865286089267954139359e-01 1.51407479084209092151658282432436e-02
            2.49614419434176670886671445259708e-01 1.03589969002955084587114242822281e-01 1.90820321320367032591480693781705e-02
            1.03589967812852373763021773811488e-01 6.46795611281056226360419714183081e-01 1.90820320200718561431951769691295e-02
            6.46795616892277380749476378696272e-01 2.49614415590636157249448956463311e-01 1.90820317414477809425665810749706e-02
            1.02029245899726006552299395480077e-01 2.76888289616485705479931311856490e-01 1.90996691614169246398180490587038e-02
            2.76888294228863351964520234105294e-01 6.21082463846217014413753076951252e-01 1.90996686937001634853938725200351e-02
            6.21082464001205591941356942697894e-01 1.02029246368156176205665985889937e-01 1.90996692404052093028354875059449e-02
            2.26789276022037022961796992603922e-01 2.33728374217554740743807428771106e-01 2.04235756492426485941837910331742e-02
            2.33728371859891315143187284775195e-01 5.39482360682531081330637334758649e-01 2.04235749257686768909358931978204e-02
            5.39482355318465778282188693992794e-01 2.26789276022037022961796992603922e-01 2.04235750270172211617847324305330e-02
            4.37409498131037799417697442549979e-01 9.59920858845616786769738837392651e-02 2.05428460504838393541060526104047e-02
            9.59920836950179934277116444718558e-02 4.66598416159025475291599605043302e-01 2.05428456208598293752931596145572e-02
            4.66598421189779177797873899180559e-01 4.37409494682125088438340299035190e-01 2.05428457919928607322557212455649e-02
            3.80758577881083304372822340155835e-01 2.27855355201996984115808686510718e-01 2.46283800576019562544871632780996e-02
            2.27855351201257216864348720264388e-01 3.91386077299870405177273369190516e-01 2.46283803633365848406722165009342e-02
            3.91386069899954280870701950334478e-01 3.80758578955004756139146593341138e-01 2.46283798539006330841072411885762e-02
            ];
        
    case  14
        
        % ALG. DEG.:   14
        % PTS CARD.:   40
        % NEG. W.  :    0
        % OUT PTS. :    3
        % M.E.INF. : 9.5e-16
        
        xyw=[
            3.33333333333333092785011331216083e-01 3.33333333333333592385372412536526e-01 8.60977427170038847603628084925731e-03
            1.22379568942328237346472974422795e-01 -2.21163705785205715048657992838343e-02 6.11721501905188708679561582215456e-04
            -2.21163705785206478326987422633465e-02 8.99736801636192295994476353371283e-01 6.11721501905184697131523385849050e-04
            8.99736801636192518039081278402591e-01 1.22379568942328126324170511907141e-01 6.11721501905187407636954599610135e-04
            2.56919955958364865100662655095221e-01 7.42961907725503301058722627203679e-01 2.81290396466153452478820007343074e-03
            7.42961907725503079014117702172371e-01 1.18136316131866407337473734440181e-04 2.81290396466155100466122185309814e-03
            1.18136316131851580872764995167046e-04 2.56919955958365364701023736415664e-01 2.81290396466155057098035285889637e-03
            2.24127192652311194032144214816071e-02 2.12963622799149272790675269106941e-02 3.09017784895421893098799159815826e-03
            2.12963622799148960540449593281664e-02 9.56290918454854033114997946540825e-01 3.09017784895421329313669467353520e-03
            9.56290918454854033114997946540825e-01 2.24127192652310812392979499918511e-02 3.09017784895421372681756366773698e-03
            1.07983678184864761995243043202208e-01 8.73605408045508569081505356734851e-01 5.47885183869874312773129432230235e-03
            8.73605408045508458059202894219197e-01 1.84109137696266932093802637382396e-02 5.47885183869874312773129432230235e-03
            1.84109137696267001482741676454680e-02 1.07983678184864984039847968233516e-01 5.47885183869874226036955633389880e-03
            4.54399154241966474199188041893649e-01 8.40211993845903226674209918201086e-03 5.73117112115491701002101265771671e-03
            8.40211993845906175704119078773147e-03 5.37198725819574396389555204223143e-01 5.73117112115491961210622662292735e-03
            5.37198725819574507411857666738797e-01 4.54399154241966474199188041893649e-01 5.73117112115491093848884673889188e-03
            2.60937045453201454936476011425839e-01 2.57352549212651951493491253586399e-02 1.11478907448084047643321881082557e-02
            2.57352549212651743326674136369547e-02 7.13327699625533329097493151493836e-01 1.11478907448084064990556640850627e-02
            7.13327699625533329097493151493836e-01 2.60937045453201454936476011425839e-01 1.11478907448084030296087121314486e-02
            1.08651864883571383502136598053767e-01 4.73180071101256538579349353312864e-02 1.16914330405722202699259071323468e-02
            4.73180071101256261023593197023729e-02 8.44030128006302948762140658800490e-01 1.16914330405722115963085272483113e-02
            8.44030128006303059784443121316144e-01 1.08651864883571369624348790239310e-01 1.16914330405722168004789551787326e-02
            3.63151187936801678191045539279003e-01 6.00340343076415372536303038941696e-01 1.32790329034408929181898528781858e-02
            6.00340343076415261514000576426042e-01 3.65084689867830117226965569443564e-02 1.32790329034409050612541847158354e-02
            3.65084689867830256004843647588132e-02 3.63151187936801733702196770536830e-01 1.32790329034408929181898528781858e-02
            1.88070100346088509013497969135642e-01 7.41933483171285490520574512629537e-01 1.54252423911586476062973716238957e-02
            7.41933483171285490520574512629537e-01 6.99964164826261114882299807504751e-02 1.54252423911586476062973716238957e-02
            6.99964164826261114882299807504751e-02 1.88070100346088509013497969135642e-01 1.54252423911586528104677995543170e-02
            3.93963054316368299634376626272569e-01 8.32192508939270103596186345384922e-02 2.03405252105097328341187790101685e-02
            8.32192508939270658707698657963192e-02 5.22817694789704745517155970446765e-01 2.03405252105097467119065868246253e-02
            5.22817694789704634494853507931111e-01 3.93963054316368355145527857530396e-01 2.03405252105097363035657309637827e-02
            2.04286459600112130230442630818288e-01 1.37280352612603406825542151636910e-01 2.18261971804863154356102228348391e-02
            1.37280352612603379069966536007996e-01 6.58433187787284657233044526947197e-01 2.18261971804863119661632708812249e-02
            6.58433187787284546210742064431543e-01 2.04286459600112074719291399560461e-01 2.18261971804863015578224150203823e-02
            3.15511927618559784303187143450486e-01 5.33376563749646703804785374813946e-01 2.54468684897090971497313205418322e-02
            5.33376563749646703804785374813946e-01 1.51111508631793539647603097364481e-01 2.54468684897090936802843685882181e-02
            1.51111508631793595158754328622308e-01 3.15511927618559839814338374708313e-01 2.54468684897090971497313205418322e-02
            3.24023305595123767286480642724200e-01 2.28102761121843300440303892173688e-01 2.69147256733732846700757335156595e-02
            2.28102761121843383707030739060428e-01 4.47873933283033043295517927617766e-01 2.69147256733732569145001178867460e-02
            4.47873933283032876762064233844285e-01 3.24023305595123767286480642724200e-01 2.69147256733733020173104932837305e-02
            ];
        
    case  15
        
        % ALG. DEG.:   15
        % PTS CARD.:   46
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.0e-16
        
        xyw=[
            3.33333292261711877646490620463737e-01 3.33333369225872422347833889944013e-01 2.33748155875758353849924731093779e-02
            1.84348340534850607674854927608976e-02 2.10206116064859810377285498361744e-02 2.51393589507226860105260080047174e-03
            2.10206198587913860309495106548638e-02 9.60544553925707256247790155612165e-01 2.51393569637528395410708270674149e-03
            9.60544579872547044097075286117615e-01 1.84348154031861895252308869430635e-02 2.51393239452845198081409883172910e-03
            9.44694948714600857275769385523745e-02 1.40084791735677512136692968169882e-02 3.79933836817317963779894185449848e-03
            1.40084495619432081864319528108354e-02 8.91522111110737869310582937032450e-01 3.79932899672832252724319346270931e-03
            8.91522077853913352818437942914898e-01 9.44693784006580272727404690158437e-02 3.79934737306244184329440116698606e-03
            2.21059623335564947588238737807842e-01 1.46001805115354987840703415713506e-02 5.36606624325662064090014524708749e-03
            1.46001627401248310761383564226890e-02 7.64340360199224910786597320111468e-01 5.36605572563657318152507968989084e-03
            7.64340671414178274289952241815627e-01 2.21059339300117646542886973293207e-01 5.36598839236875936598014646961019e-03
            1.83161349800822388644849780803270e-02 1.06831540445691930418803394786664e-01 5.41543061174039330740725972646032e-03
            1.06831572649002698449649528811278e-01 8.74852289591818377267884443426738e-01 5.41543292052138088010782013270727e-03
            8.74852363873429461627040382154519e-01 1.83161445101451825223648484097794e-02 5.41543076261086220313245576107875e-03
            3.84111880036755948264470816866378e-01 1.35065559081281177933320236661530e-02 6.12435944557631825602150854592765e-03
            1.35065505592318075428481805033698e-02 6.02381757760965697023891607386759e-01 6.12435703950236273035434564349089e-03
            6.02381907561370333326067338930443e-01 3.84111551382989191427697051040013e-01 6.12435463564138353464993969055286e-03
            1.76095784933983541542534823065580e-02 2.46869564037977468062834418560669e-01 7.19892017990240648539490564417065e-03
            2.46869635891989791520728658724693e-01 7.35520785115458264336041338538053e-01 7.19892224504508487781251346859790e-03
            7.35520958397237634152077134785941e-01 1.76095847380971885776101970577656e-02 7.19892013179164024644363095717381e-03
            1.62110954488718303490024652546708e-02 4.20027449315563017329111517028650e-01 7.49190020798990233275027605941432e-03
            4.20027557266491768928062811028212e-01 5.63761341880282595973028492153389e-01 7.49190393884934403151421022926115e-03
            5.63761608884686626907978279632516e-01 1.62111046483847491550545072414025e-02 7.49190232121703901524423230284810e-03
            9.45835540958728515636977363101323e-02 7.81937423335833381443649159336928e-02 9.61633693621145001217964676243355e-03
            7.81936458310440452024181468004826e-02 8.27222782704063241965286579215899e-01 9.61633495119591438782258308037854e-03
            8.27222507563532660768146342888940e-01 9.45835855871118430515664954327804e-02 9.61634719828197842561579022913065e-03
            2.23020503669574404748132678832917e-01 6.90247082160969777087089482847659e-02 1.04173339778732328836374776415141e-02
            6.90245901886988449946969126358454e-02 7.07954806600517905224023706978187e-01 1.04173200840826374324876724131173e-02
            7.07955491881565479594939915841678e-01 2.23020657547390782626806071675674e-01 1.04172557218891737029764499311568e-02
            9.13686563942972540530718106310815e-02 1.97441541806721687635217676870525e-01 1.42884849730802853534639140775653e-02
            1.97441432426123636423298535191861e-01 7.11189907126281162241809852275765e-01 1.42884846869895058146804345255987e-02
            7.11189569389398590359974150487687e-01 9.13686840082226642278939721109055e-02 1.42884890448401660062449636257043e-02
            3.84530855617108480171140172387823e-01 7.30977274485456463004595661914209e-02 1.45158335893254743925728789122331e-02
            7.30977106499786088988912524655461e-02 5.42371300752677920620214990776731e-01 1.45158336691942786711440049884914e-02
            5.42371248933846916706613683345495e-01 3.84531035703955026416878126838128e-01 1.45158423302523161002586249423985e-02
            2.15038812389831607285728409806325e-01 1.55893061854053271986231266055256e-01 1.51201380421244653551315906270247e-02
            1.55892843158961874605594744025439e-01 6.29068363385964834932906342146453e-01 1.51201294906665728579042706769542e-02
            6.29068887420656586151324063393986e-01 2.15038957479725206800580394883582e-01 1.51202112131431642971790552110178e-02
            8.53189508964801446300185716609121e-02 3.59007499441052202460156195229501e-01 1.62557619200069045084511287768692e-02
            3.59007386827242869209442233113805e-01 5.55673632624283819758659319631988e-01 1.62557683569843729187720526851990e-02
            5.55673357289606983933083483861992e-01 8.53189724946177840969596672948683e-02 1.62557597236451592537243016067805e-02
            2.04230402773116243020368187899294e-01 2.86157186802203589159887542336946e-01 2.00935650478077099112539372072206e-02
            2.86156970552324307899283439837745e-01 5.09612567484152623187299013807205e-01 2.00935725508967488739386908491724e-02
            5.09612717585778640660976179788122e-01 2.04230501683514309441136447276222e-01 2.00936262301876777947029495408060e-02
            3.63891035835017306876437714890926e-01 1.81085979988352741010260160692269e-01 2.06576753252571722330177550475128e-02
            1.81085896307873084332129565154901e-01 4.55023141283621535002623659238452e-01 2.06576526540188792901986403194314e-02
            4.55022970575782392810282317441306e-01 3.63891085598291641467483259475557e-01 2.06576631688787891449887723638312e-02
            ];
        
    case  16
        
        % ALG. DEG.:   16
        % PTS CARD.:   51
        % NEG. W.  :    0
        % OUT PTS. :    6
        % M.E.INF. : 1.8e-15
        
        xyw=[
            2.60234718756303995412793028663145e-01 -7.79627663672445049558579199810993e-03 1.32918295858666592364205616405570e-03
            -7.79627663672445830184143389374185e-03 7.47561557880420446409175383450929e-01 1.32918295858666223735466971334063e-03
            7.47561557880420335386872920935275e-01 2.60234718756303995412793028663145e-01 1.32918295858666700784422864956014e-03
            8.76543641109690518931074620923027e-02 9.09443987549238164191933719848748e-01 1.69980304133707476980930106691403e-03
            9.09443987549238164191933719848748e-01 2.90164833979279258857619794298444e-03 1.69980304133707932345842550603265e-03
            2.90164833979275052153190550541240e-03 8.76543641109690102597440386489325e-02 1.69980304133707650453277704372113e-03
            3.00672736234540771693701799449627e-01 7.00388362907983452743110319715925e-01 1.75956144376605383254641434831456e-03
            7.00388362907983674787715244747233e-01 -1.06109914252419733175780702794100e-03 1.75956144376605947039771127293761e-03
            -1.06109914252421858212038774382791e-03 3.00672736234540771693701799449627e-01 1.75956144376605448306771783961722e-03
            1.88499527230496406737803027908740e-02 1.64729254038669808979378217372869e-02 2.02207213839060732582519896993745e-03
            1.64729254038669566118091580619875e-02 9.64677121873083409653304443054367e-01 2.02207213839060645846346098153390e-03
            9.64677121873083409653304443054367e-01 1.88499527230495955709699273938895e-02 2.02207213839060906054867494674454e-03
            9.84167329545414665270541831887385e-02 1.19409662687472318404191540253123e-02 3.41472365663329302867823322742424e-03
            1.19409662687472370445895819557336e-02 8.89642300776711358878401370020583e-01 3.41472365663328782450780529700296e-03
            8.89642300776711358878401370020583e-01 9.84167329545414526492663753742818e-02 3.41472365663328218665650837237990e-03
            4.60403420093106141575134415688808e-01 1.17880011749511917629185120404145e-02 5.22009725933132804481795474771388e-03
            1.17880011749511449253846606666229e-02 5.27808578731942534822962898033438e-01 5.22009725933132804481795474771388e-03
            5.27808578731942423800660435517784e-01 4.60403420093106363619739340720116e-01 5.22009725933133238162664468973162e-03
            7.38297774166404574547328820699477e-02 8.68681356668872584059215569141088e-01 7.36534479219431905544501759663945e-03
            8.68681356668872473036913106625434e-01 5.74888659144870209360966839540197e-02 7.36534479219432686170065949227137e-03
            5.74888659144869793027332605106494e-02 7.38297774166404852103084976988612e-02 7.36534479219431992280675558504299e-03
            1.86109588791245589289857775838755e-01 7.87963168691664939125018918275600e-01 7.49859320680401359587863296951582e-03
            7.87963168691665050147321380791254e-01 2.59272425170894195434190265814323e-02 7.49859320680400492226125308548035e-03
            2.59272425170893917878434109525188e-02 1.86109588791245617045433391467668e-01 7.49859320680400752434646705069099e-03
            3.83201939072719266476951816002838e-01 5.88557717697621440855471064423909e-01 8.28159353952347999927585675550290e-03
            5.88557717697621218810866139392601e-01 2.82403432296592579731076000371104e-02 8.28159353952347306038195284827452e-03
            2.82403432296592718508954078515671e-02 3.83201939072719321988103047260665e-01 8.28159353952347479510542882508162e-03
            1.86107863137863926983328610731405e-01 4.21292162463990718657136369529326e-02 9.87408590048508352787770547820401e-03
            4.21292162463990441101380213240191e-02 7.71762920615737035845427271851804e-01 9.87408590048508699732465743181820e-03
            7.71762920615737035845427271851804e-01 1.86107863137863926983328610731405e-01 9.87408590048508352787770547820401e-03
            3.29435605315130441983484388401848e-01 3.83979662592876733184787951813632e-02 1.06015728208726772024972362373774e-02
            3.83979662592876594406909873669065e-02 6.32166428425581905514718528138474e-01 1.06015728208726806719441881909916e-02
            6.32166428425581794492416065622820e-01 3.29435605315130497494635619659675e-01 1.06015728208726754677737602605703e-02
            2.72057145537797639978094821344712e-01 6.51051865673838658921113164979033e-01 1.22243292626775211517919927928233e-02
            6.51051865673838547898810702463379e-01 7.68909887883636317118529746039712e-02 1.22243292626775280906858967000517e-02
            7.68909887883636733452163980473415e-02 2.72057145537797639978094821344712e-01 1.22243292626775142128980888855949e-02
            1.41235187360025743474878368033387e-01 7.29399431838533618410735925863264e-01 1.40749434644868894050961927177923e-02
            7.29399431838533618410735925863264e-01 1.29365380801440610358810090474435e-01 1.40749434644868980787135726018278e-02
            1.29365380801440610358810090474435e-01 1.41235187360025798986029599291214e-01 1.40749434644868980787135726018278e-02
            4.49015766811866112728779398821644e-01 9.76217927915696837004588815034367e-02 1.84510672837505315668771288528660e-02
            9.76217927915697392116101127612637e-02 4.53362440396564259081912950932747e-01 1.84510672837505350363240808064802e-02
            4.53362440396564259081912950932747e-01 4.49015766811866057217628167563817e-01 1.84510672837505350363240808064802e-02
            2.66798474540568653079475325284875e-01 1.37399173718296491175649975957640e-01 1.85238743065528205233860603584617e-02
            1.37399173718296463420074360328726e-01 5.95802351741134939011601545644226e-01 1.85238743065528170539391084048475e-02
            5.95802351741134939011601545644226e-01 2.66798474540568653079475325284875e-01 1.85238743065528205233860603584617e-02
            2.82382350376655322232721800901345e-01 5.34917405870669493417324247275246e-01 2.06662750426762041600081687420243e-02
            5.34917405870669382395021784759592e-01 1.82700243752675295372256414339063e-01 2.06662750426761972211142648347959e-02
            1.82700243752675239861105183081236e-01 2.82382350376655322232721800901345e-01 2.06662750426762041600081687420243e-02
            3.49747383256928001671326455834787e-01 2.38132117614774041491898515232606e-01 2.36595465485981044928553984618702e-02
            2.38132117614773985980747283974779e-01 4.12120499128298012347926260190434e-01 2.36595465485981079623023504154844e-02
            4.12120499128298067859077491448261e-01 3.49747383256928001671326455834787e-01 2.36595465485980940845145426010276e-02
            ];
        
    case  17
        
        % ALG. DEG.:   17
        % PTS CARD.:   57
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.8e-16
        
        xyw=[
            1.47209661956265915744301864265253e-02 1.76399848555067323419009994722728e-02 1.68851800404976482711250085344545e-03
            1.76399848555067323419009994722728e-02 9.67639048948866653532263626402710e-01 1.68851800404976482711250085344545e-03
            9.67639048948866653532263626402710e-01 1.47209661956265915744301864265253e-02 1.68851800404976482711250085344545e-03
            7.58278723376997892957618319087487e-02 7.69747584377225568808000844001072e-03 1.90996283509696517834142159841804e-03
            7.69747584377225568808000844001072e-03 9.16474651818527941138370351836784e-01 1.90996283509696517834142159841804e-03
            9.16474651818527941138370351836784e-01 7.58278723376997892957618319087487e-02 1.90996283509696517834142159841804e-03
            3.10585516558464003189499180734856e-01 8.32009195104604741544118695628640e-03 3.31201304072067459829953328664942e-03
            8.32009195104604741544118695628640e-03 6.81094391490490003171487387589877e-01 3.31201304072067459829953328664942e-03
            6.81094391490490003171487387589877e-01 3.10585516558464003189499180734856e-01 3.31201304072067459829953328664942e-03
            1.44883146249404001770777483670827e-02 9.01640395800713256857505939478870e-02 3.64245036743674052356900538995887e-03
            9.01640395800713256857505939478870e-02 8.95347645794988267198277753777802e-01 3.64245036743674052356900538995887e-03
            8.95347645794988267198277753777802e-01 1.44883146249404001770777483670827e-02 3.64245036743674052356900538995887e-03
            1.78143852253019957521118499244039e-01 1.64238580465156137899729316131925e-02 4.98250123670270373549273656976766e-03
            1.64238580465156137899729316131925e-02 8.05432289700464432158355521096382e-01 4.98250123670270373549273656976766e-03
            8.05432289700464432158355521096382e-01 1.78143852253019957521118499244039e-01 4.98250123670270373549273656976766e-03
            4.66133550795934714638946161358035e-01 1.23355817687377852104324560400528e-02 5.13817254929797503915089862402965e-03
            1.23355817687377852104324560400528e-02 5.21530867435327394332489348016679e-01 5.13817254929797503915089862402965e-03
            5.21530867435327394332489348016679e-01 4.66133550795934714638946161358035e-01 5.13817254929797503915089862402965e-03
            1.47365661662822756972079929482788e-02 2.09325171157457634585341565980343e-01 5.15474565188859049080782881446794e-03
            2.09325171157457634585341565980343e-01 7.75938262676260093186897393024992e-01 5.15474565188859049080782881446794e-03
            7.75938262676260093186897393024992e-01 1.47365661662822756972079929482788e-02 5.15474565188859049080782881446794e-03
            1.43325355478803179165581127563200e-02 3.58602315795423676192399398132693e-01 5.80111456840011127655820644122286e-03
            3.58602315795423676192399398132693e-01 6.27065148656696047524405912554357e-01 5.80111456840011127655820644122286e-03
            6.27065148656696047524405912554357e-01 1.43325355478803179165581127563200e-02 5.80111456840011127655820644122286e-03
            7.53033493906421630414982359980058e-02 5.59681329947420772019661683316372e-02 6.23487189374104591732539404347335e-03
            5.59681329947420772019661683316372e-02 8.68728517614615669550914844876388e-01 6.23487189374104591732539404347335e-03
            8.68728517614615669550914844876388e-01 7.53033493906421630414982359980058e-02 6.23487189374104591732539404347335e-03
            3.17339393501727851099047938987496e-01 5.16948784187994947458832939446438e-02 9.81623975776597587628025110007002e-03
            5.16948784187994947458832939446438e-02 6.30965728079472598643917535810033e-01 9.81623975776597587628025110007002e-03
            6.30965728079472598643917535810033e-01 3.17339393501727851099047938987496e-01 9.81623975776597587628025110007002e-03
            7.53542654903545661149166789982701e-02 1.54598211665730095321080739267927e-01 1.00507535375725948895153649687018e-02
            1.54598211665730095321080739267927e-01 7.70047522843915421830729428620543e-01 1.00507535375725948895153649687018e-02
            7.70047522843915421830729428620543e-01 7.53542654903545661149166789982701e-02 1.00507535375725948895153649687018e-02
            1.82570774440876770761121861141874e-01 2.66530059635146754359880105766933e-01 1.04530227842825687578631743690494e-02
            2.66530059635146754359880105766933e-01 5.50899165923976474878998033091193e-01 1.04530227842825687578631743690494e-02
            5.50899165923976474878998033091193e-01 1.82570774440876770761121861141874e-01 1.04530227842825687578631743690494e-02
            1.79710032722074752875585090805544e-01 8.15862277549058850789620578325412e-02 1.06661238358018613248123784842392e-02
            8.15862277549058850789620578325412e-02 7.38703739523019375923240659176372e-01 1.06661238358018613248123784842392e-02
            7.38703739523019375923240659176372e-01 1.79710032722074752875585090805544e-01 1.06661238358018613248123784842392e-02
            1.75807941715677179539412122721842e-01 1.80353640520785951650850620353594e-01 1.18746885945923402044810401889663e-02
            1.80353640520785951650850620353594e-01 6.43838417763536896565312872553477e-01 1.18746885945923402044810401889663e-02
            6.43838417763536896565312872553477e-01 1.75807941715677179539412122721842e-01 1.18746885945923402044810401889663e-02
            4.58731499688783517232337771929451e-01 4.74126007719326725275976741613704e-01 1.23885120040660139800747785443491e-02
            4.74126007719326725275976741613704e-01 6.71424925918898130028367177146720e-02 1.23885120040660139800747785443491e-02
            6.71424925918898130028367177146720e-02 4.58731499688783517232337771929451e-01 1.23885120040660139800747785443491e-02
            7.53196920743743475457065983391658e-02 2.93754105362967754544456511212047e-01 1.27927851211363120126351944350063e-02
            2.93754105362967754544456511212047e-01 6.30926202562657856276473467005417e-01 1.27927851211363120126351944350063e-02
            6.30926202562657856276473467005417e-01 7.53196920743743475457065983391658e-02 1.27927851211363120126351944350063e-02
            1.63305177925384792381180432130350e-01 3.84251187765941693008642232598504e-01 1.56554294016505793352145303742873e-02
            3.84251187765941693008642232598504e-01 4.52443634308673514610177335271146e-01 1.56554294016505793352145303742873e-02
            4.52443634308673514610177335271146e-01 1.63305177925384792381180432130350e-01 1.56554294016505793352145303742873e-02
            3.11813568699614629409211374877486e-01 1.39501525996608172208723885887593e-01 1.57951812747187600449727540308231e-02
            1.39501525996608172208723885887593e-01 5.48684905303777226137640354863834e-01 1.57951812747187600449727540308231e-02
            5.48684905303777226137640354863834e-01 3.11813568699614629409211374877486e-01 1.57951812747187600449727540308231e-02
            3.06048259112027931649890888365917e-01 2.67720653545962083974529832630651e-01 1.93095802077451229872284699240481e-02
            2.67720653545962083974529832630651e-01 4.26231087342009984375579279003432e-01 1.93095802077451229872284699240481e-02
            4.26231087342009984375579279003432e-01 3.06048259112027931649890888365917e-01 1.93095802077451229872284699240481e-02
            ];
        
    case  18
        
        % ALG. DEG.:   18
        % PTS CARD.:   64
        % NEG. W.  :    1
        % OUT PTS. :    0
        % M.E.INF. : 4.9e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 -6.29226107924366201418209243456658e-02
            9.22437177016372997218063289892598e-03 9.54037638902169304877531885722419e-03 6.48635552623456597756690378275835e-04
            9.54037638902169304877531885722419e-03 9.81235251840814504120658057217952e-01 6.48635552623456597756690378275835e-04
            9.81235251840814504120658057217952e-01 9.22437177016372997218063289892598e-03 6.48635552623456597756690378275835e-04
            1.15881748682225369645459878142901e-02 5.99102344454169738474647033399378e-02 2.20217225795283884706710075818137e-03
            5.99102344454169738474647033399378e-02 9.28501590686360511739394496544264e-01 2.20217225795283884706710075818137e-03
            9.28501590686360511739394496544264e-01 1.15881748682225369645459878142901e-02 2.20217225795283884706710075818137e-03
            1.37109449232486801939501219749218e-01 8.13423222383992221284909618361780e-03 2.27690138938589333242323853312428e-03
            8.13423222383992221284909618361780e-03 8.54756318543673287990714015904814e-01 2.27690138938589333242323853312428e-03
            8.54756318543673287990714015904814e-01 1.37109449232486801939501219749218e-01 2.27690138938589333242323853312428e-03
            5.40746581302322046336250593867589e-02 1.59640348658512051804869713578228e-02 2.62610584374150385356116998991638e-03
            1.59640348658512051804869713578228e-02 9.29961307003916659574827008327702e-01 2.62610584374150385356116998991638e-03
            9.29961307003916659574827008327702e-01 5.40746581302322046336250593867589e-02 2.62610584374150385356116998991638e-03
            1.23562085317423936414726881594106e-02 1.56321458214785352636155835170939e-01 3.68731893610492553886581745814510e-03
            1.56321458214785352636155835170939e-01 8.31322333253472267600159284484107e-01 3.68731893610492553886581745814510e-03
            8.31322333253472267600159284484107e-01 1.23562085317423936414726881594106e-02 3.68731893610492553886581745814510e-03
            2.54443010786652157317888622856117e-01 1.32280360805979747829130843683743e-02 4.41739343629661811857500808287114e-03
            1.32280360805979747829130843683743e-02 7.32328953132749815857494013471296e-01 4.41739343629661811857500808287114e-03
            7.32328953132749815857494013471296e-01 2.54443010786652157317888622856117e-01 4.41739343629661811857500808287114e-03
            1.22006545481682538573098284473417e-02 4.38518077301824849012490403765696e-01 4.82388821244136584837081116461377e-03
            4.38518077301824849012490403765696e-01 5.49281268150006996009437898464967e-01 4.82388821244136584837081116461377e-03
            5.49281268150006996009437898464967e-01 1.22006545481682538573098284473417e-02 4.82388821244136584837081116461377e-03
            3.95592717135403115769065607310040e-01 1.29052660579212063324261450247832e-02 4.93354989021067228543993010703161e-03
            1.29052660579212063324261450247832e-02 5.91502016806675690041572579502827e-01 4.93354989021067228543993010703161e-03
            5.91502016806675690041572579502827e-01 3.95592717135403115769065607310040e-01 4.93354989021067228543993010703161e-03
            1.39894903845946654008525200652002e-02 2.87578430196016754383236957437475e-01 5.13035659461587214419386882013896e-03
            2.87578430196016754383236957437475e-01 6.98432079419388562868675762729254e-01 5.13035659461587214419386882013896e-03
            6.98432079419388562868675762729254e-01 1.39894903845946654008525200652002e-02 5.13035659461587214419386882013896e-03
            6.12426222536310363442346726969845e-02 8.21310314077598468962548849958694e-02 5.89834799153866190524198032107961e-03
            8.21310314077598468962548849958694e-02 8.56626346338609012676101883698720e-01 5.89834799153866190524198032107961e-03
            8.56626346338609012676101883698720e-01 6.12426222536310363442346726969845e-02 5.89834799153866190524198032107961e-03
            1.40187294281714058818977264309069e-01 5.09884646182077044818115041380224e-02 6.49275376755569823272740137554138e-03
            5.09884646182077044818115041380224e-02 8.08824241100078222821423423738452e-01 6.49275376755569823272740137554138e-03
            8.08824241100078222821423423738452e-01 1.40187294281714058818977264309069e-01 6.49275376755569823272740137554138e-03
            6.59534606576724913296061458822805e-02 1.82306457551239170511436782362580e-01 8.15910291228638846761089808978795e-03
            1.82306457551239170511436782362580e-01 7.51740081791088310403381456126226e-01 8.15910291228638846761089808978795e-03
            7.51740081791088310403381456126226e-01 6.59534606576724913296061458822805e-02 8.15910291228638846761089808978795e-03
            6.24139442088075535597369025708758e-02 4.11715469349845597513137818168616e-01 8.76280825311153907519745587251236e-03
            4.11715469349845597513137818168616e-01 5.25870586441346876682700894889422e-01 8.76280825311153907519745587251236e-03
            5.25870586441346876682700894889422e-01 6.24139442088075535597369025708758e-02 8.76280825311153907519745587251236e-03
            7.43017152789951784708222248809761e-02 2.93703944612865197605344746989431e-01 9.20591572143557822227588616215144e-03
            2.93703944612865197605344746989431e-01 6.31994340108139596168257412500679e-01 9.20591572143557822227588616215144e-03
            6.31994340108139596168257412500679e-01 7.43017152789951784708222248809761e-02 9.20591572143557822227588616215144e-03
            2.55758429076780702260407451831270e-01 6.81831468635734766481348856359546e-02 9.53448529703253451483835334556716e-03
            6.81831468635734766481348856359546e-02 6.76058424059645779458094239089405e-01 9.53448529703253451483835334556716e-03
            6.76058424059645779458094239089405e-01 2.55758429076780702260407451831270e-01 9.53448529703253451483835334556716e-03
            3.89748898002750199243138240490225e-01 6.69978935694291871660865922422090e-02 1.02910923429764718239898257934328e-02
            6.69978935694291871660865922422090e-02 5.43253208427820655224138590710936e-01 1.02910923429764718239898257934328e-02
            5.43253208427820655224138590710936e-01 3.89748898002750199243138240490225e-01 1.02910923429764718239898257934328e-02
            1.55168432990029553364408343441028e-01 1.40086682350669539109944139454456e-01 1.17945706246527434735149597599957e-02
            1.40086682350669539109944139454456e-01 7.04744884659300852014496285846690e-01 1.17945706246527434735149597599957e-02
            7.04744884659300852014496285846690e-01 1.55168432990029553364408343441028e-01 1.17945706246527434735149597599957e-02
            1.70472927860750189665850484743714e-01 2.61254002060855416988971455793944e-01 1.55454761315940367949472999953286e-02
            2.61254002060855416988971455793944e-01 5.68273070078394448856329290720169e-01 1.55454761315940367949472999953286e-02
            5.68273070078394448856329290720169e-01 1.70472927860750189665850484743714e-01 1.55454761315940367949472999953286e-02
            2.86469196549182070210548545219353e-01 1.63053894264229143340472205636615e-01 1.60199834936127216700185726949712e-02
            1.63053894264229143340472205636615e-01 5.50476909186588758693403633515118e-01 1.60199834936127216700185726949712e-02
            5.50476909186588758693403633515118e-01 2.86469196549182070210548545219353e-01 1.60199834936127216700185726949712e-02
            1.55079032148931039714057078526821e-01 4.08496920155350773384839158097748e-01 1.63302088864774180843042472588422e-02
            4.08496920155350773384839158097748e-01 4.36424047695718186901103763375431e-01 1.63302088864774180843042472588422e-02
            4.36424047695718186901103763375431e-01 1.55079032148931039714057078526821e-01 1.63302088864774180843042472588422e-02
            3.06409473651319375075985362855135e-01 3.04333597697522717506757317096344e-01 3.88598027284986005658318219957437e-02
            3.04333597697522717506757317096344e-01 3.89256928651157851906106088790693e-01 3.88598027284986005658318219957437e-02
            3.89256928651157851906106088790693e-01 3.06409473651319375075985362855135e-01 3.88598027284986005658318219957437e-02
            ];
        
    case  19
        
        % ALG. DEG.:   19
        % PTS CARD.:   70
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.2e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 1.52314808800737087196575103575924e-02
            1.03766565457017543444573348665472e-02 1.44810472658998996842161233189472e-02 9.75090979032849436772267814887982e-04
            1.44810472658998996842161233189472e-02 9.75142296188398316481027450208785e-01 9.75090979032849436772267814887982e-04
            9.75142296188398316481027450208785e-01 1.03766565457017543444573348665472e-02 9.75090979032849436772267814887982e-04
            5.28956402129591926608576102353254e-02 8.92711734256453406932685368246894e-03 1.38097230858658754845680327605351e-03
            8.92711734256453406932685368246894e-03 9.38177242444476200411429545056308e-01 1.38097230858658754845680327605351e-03
            9.38177242444476200411429545056308e-01 5.28956402129591926608576102353254e-02 1.38097230858658754845680327605351e-03
            1.30940571465279348428722983044281e-01 6.39606356170804597055523643689412e-03 1.76658971586647118613699891653823e-03
            6.39606356170804597055523643689412e-03 8.62663364973012547487485335295787e-01 1.76658971586647118613699891653823e-03
            8.62663364973012547487485335295787e-01 1.30940571465279348428722983044281e-01 1.76658971586647118613699891653823e-03
            1.26681865510903015475951960411294e-02 7.29082350599041251193455082102446e-02 2.57601453884506942718091160315907e-03
            7.29082350599041251193455082102446e-02 9.14423578389005564659441915864591e-01 2.57601453884506942718091160315907e-03
            9.14423578389005564659441915864591e-01 1.26681865510903015475951960411294e-02 2.57601453884506942718091160315907e-03
            2.46613960104767354630794784497994e-01 1.13210284482716614512520791890893e-02 3.65585508313527205934145847265881e-03
            1.13210284482716614512520791890893e-02 7.42065011446961020347146131825866e-01 3.65585508313527205934145847265881e-03
            7.42065011446961020347146131825866e-01 2.46613960104767354630794784497994e-01 3.65585508313527205934145847265881e-03
            1.24330000962839457012698574089882e-02 1.71507203857275219993638870619179e-01 3.69473652588097082813400895417999e-03
            1.71507203857275219993638870619179e-01 8.16059796046440855121772983693518e-01 3.69473652588097082813400895417999e-03
            8.16059796046440855121772983693518e-01 1.24330000962839457012698574089882e-02 3.69473652588097082813400895417999e-03
            9.45870867926038548512135406554080e-03 4.54553083656948797575125809089513e-01 3.77389302633806883111033947386659e-03
            4.54553083656948797575125809089513e-01 5.35988207663790849899498880404280e-01 3.77389302633806883111033947386659e-03
            5.35988207663790849899498880404280e-01 9.45870867926038548512135406554080e-03 3.77389302633806883111033947386659e-03
            3.82677127432900721082376094273059e-01 1.03041271596045185682255507231275e-02 3.80773420333801061038458968255327e-03
            1.03041271596045185682255507231275e-02 6.07018745407494786370250494655920e-01 3.80773420333801061038458968255327e-03
            6.07018745407494786370250494655920e-01 3.82677127432900721082376094273059e-01 3.80773420333801061038458968255327e-03
            1.07221902311333636831358617769183e-02 3.02819418977863463737776328343898e-01 3.95037906656455200388178283787965e-03
            3.02819418977863463737776328343898e-01 6.86458390791003125741553958505392e-01 3.95037906656455200388178283787965e-03
            6.86458390791003125741553958505392e-01 1.07221902311333636831358617769183e-02 3.95037906656455200388178283787965e-03
            6.34588872173355300621011565453955e-02 4.79826825588268238154121547722752e-02 4.10521976646200867366864883933886e-03
            4.79826825588268238154121547722752e-02 8.88558430223837625305804976960644e-01 4.10521976646200867366864883933886e-03
            8.88558430223837625305804976960644e-01 6.34588872173355300621011565453955e-02 4.10521976646200867366864883933886e-03
            1.49585341667801352549815874226624e-01 4.15353642843105178017459877537476e-02 5.31583329040207202415002640805142e-03
            4.15353642843105178017459877537476e-02 8.08879294047888164342907657555770e-01 5.31583329040207202415002640805142e-03
            8.08879294047888164342907657555770e-01 1.49585341667801352549815874226624e-01 5.31583329040207202415002640805142e-03
            6.59383749931683749156974272409570e-02 1.27098301306499766871738188456220e-01 7.25903156102605112698356037981284e-03
            1.27098301306499766871738188456220e-01 8.06963323700331858212564384302823e-01 7.25903156102605112698356037981284e-03
            8.06963323700331858212564384302823e-01 6.59383749931683749156974272409570e-02 7.25903156102605112698356037981284e-03
            4.19360344474693613747717790829483e-01 4.84133944843579896999408163082990e-02 7.34705686595597384086486769660951e-03
            4.84133944843579896999408163082990e-02 5.32226261040948389613447488954989e-01 7.34705686595597384086486769660951e-03
            5.32226261040948389613447488954989e-01 4.19360344474693613747717790829483e-01 7.34705686595597384086486769660951e-03
            5.12859715627437540153010786525556e-02 3.85451519447595591927324676362332e-01 8.43495559667827368710568691767548e-03
            3.85451519447595591927324676362332e-01 5.63262508989660681812949860614026e-01 8.43495559667827368710568691767548e-03
            5.63262508989660681812949860614026e-01 5.12859715627437540153010786525556e-02 8.43495559667827368710568691767548e-03
            5.92161717125940026407171501432458e-02 2.44263378740914083309121451748069e-01 8.60958476231886947471760862526935e-03
            2.44263378740914083309121451748069e-01 6.96520449546491948744630917644827e-01 8.60958476231886947471760862526935e-03
            6.96520449546491948744630917644827e-01 5.92161717125940026407171501432458e-02 8.60958476231886947471760862526935e-03
            1.57744353652534891052638954533904e-01 1.07039102999093777013328576686035e-01 8.74583707273539692761410435650760e-03
            1.07039102999093777013328576686035e-01 7.35216543348371276422881237522233e-01 8.74583707273539692761410435650760e-03
            7.35216543348371276422881237522233e-01 1.57744353652534891052638954533904e-01 8.74583707273539692761410435650760e-03
            2.76356371451293159058337778333225e-01 5.93958296507460237201314612320857e-02 9.14573890014911371204586743033360e-03
            5.93958296507460237201314612320857e-02 6.64247798897960928243833222950343e-01 9.14573890014911371204586743033360e-03
            6.64247798897960928243833222950343e-01 2.76356371451293159058337778333225e-01 9.14573890014911371204586743033360e-03
            4.02925432504102898345621497355751e-01 1.11259697463848866716418228861585e-01 1.05446777512285729677676826554489e-02
            1.11259697463848866716418228861585e-01 4.85814870032048151671233426895924e-01 1.05446777512285729677676826554489e-02
            4.85814870032048151671233426895924e-01 4.02925432504102898345621497355751e-01 1.05446777512285729677676826554489e-02
            1.44741209510670071347604448419588e-01 2.07337086661877140203458225187205e-01 1.15145856655313790634087567354982e-02
            2.07337086661877140203458225187205e-01 6.47921703827452843960088557651034e-01 1.15145856655313790634087567354982e-02
            6.47921703827452843960088557651034e-01 1.44741209510670071347604448419588e-01 1.15145856655313790634087567354982e-02
            1.28002196170715176748444719123654e-01 3.38803971814336801759992567895097e-01 1.24670942256624363991823756236954e-02
            3.38803971814336801759992567895097e-01 5.33193832014948077002713944239076e-01 1.24670942256624363991823756236954e-02
            5.33193832014948077002713944239076e-01 1.28002196170715176748444719123654e-01 1.24670942256624363991823756236954e-02
            2.67334020957130047424499252883834e-01 1.47872285803578001894820204142889e-01 1.29780674745516941964718071744755e-02
            1.47872285803578001894820204142889e-01 5.84793693239291978436256158602191e-01 1.29780674745516941964718071744755e-02
            5.84793693239291978436256158602191e-01 2.67334020957130047424499252883834e-01 1.29780674745516941964718071744755e-02
            3.80640391337433547924717913701897e-01 2.07421341282564897223750222110539e-01 1.44577581531146661525388807945092e-02
            2.07421341282564897223750222110539e-01 4.11938267380001554851531864187564e-01 1.44577581531146661525388807945092e-02
            4.11938267380001554851531864187564e-01 3.80640391337433547924717913701897e-01 1.44577581531146661525388807945092e-02
            2.41019074102765712419582655456907e-01 2.67083117991290863724174187154858e-01 1.50827998399043674421227123616518e-02
            2.67083117991290863724174187154858e-01 4.91897807905943451611818773017148e-01 1.50827998399043674421227123616518e-02
            4.91897807905943451611818773017148e-01 2.41019074102765712419582655456907e-01 1.50827998399043674421227123616518e-02
            ];
        
    case  20
        
        % ALG. DEG.:   20
        % PTS CARD.:   78
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.9e-16
        
        xyw=[
            7.90368134162346309967706048382752e-03 1.08926176663298908409149845510910e-02 5.69495307448751944545684011700359e-04
            1.08926176663298908409149845510910e-02 9.81203700992046679019153998524416e-01 5.69495307448751944545684011700359e-04
            9.81203700992046679019153998524416e-01 7.90368134162346309967706048382752e-03 5.69495307448751944545684011700359e-04
            3.70261421404957180927852711249670e-03 5.68359139419127579184731757777627e-02 7.67883543729124679044772960878618e-04
            5.68359139419127579184731757777627e-02 9.39461471844037609990607506915694e-01 7.67883543729124679044772960878618e-04
            9.39461471844037609990607506915694e-01 3.70261421404957180927852711249670e-03 7.67883543729124679044772960878618e-04
            1.09337390150228475071969569398789e-01 3.03859257334367343500458602534309e-03 9.62342795893268102697115651267268e-04
            3.03859257334367343500458602534309e-03 8.87624017276427812461747635097709e-01 9.62342795893268102697115651267268e-04
            8.87624017276427812461747635097709e-01 1.09337390150228475071969569398789e-01 9.62342795893268102697115651267268e-04
            4.28810753769432936866046190971247e-02 9.80505702549880342466792626510141e-03 1.25021865617795701133840857721680e-03
            9.80505702549880342466792626510141e-03 9.47313867597557868194257935101632e-01 1.25021865617795701133840857721680e-03
            9.47313867597557868194257935101632e-01 4.28810753769432936866046190971247e-02 1.25021865617795701133840857721680e-03
            1.77473257307326614390374786012217e-03 2.36693018982774450176265190748381e-01 1.31537561809383081283530980698515e-03
            2.36693018982774450176265190748381e-01 7.61532248444152237709658948006108e-01 1.31537561809383081283530980698515e-03
            7.61532248444152237709658948006108e-01 1.77473257307326614390374786012217e-03 1.31537561809383081283530980698515e-03
            3.11814570601137097671795572750852e-01 2.54551979563882565041343930545281e-03 1.59781280701407611168074485163970e-03
            2.54551979563882565041343930545281e-03 6.85639909603224184664327367499936e-01 1.59781280701407611168074485163970e-03
            6.85639909603224184664327367499936e-01 3.11814570601137097671795572750852e-01 1.59781280701407611168074485163970e-03
            2.99000660469557651144345555849213e-03 4.63119441155936895082589899175218e-01 1.82588449703942291499347838623635e-03
            4.63119441155936895082589899175218e-01 5.33890552239367544018477929057553e-01 1.82588449703942291499347838623635e-03
            5.33890552239367544018477929057553e-01 2.99000660469557651144345555849213e-03 1.82588449703942291499347838623635e-03
            3.22142321349414423314883038074186e-02 5.24854182546855979607514086637821e-02 2.66226894560845240428337277194260e-03
            5.24854182546855979607514086637821e-02 9.15300349610372987463335903157713e-01 2.66226894560845240428337277194260e-03
            9.15300349610372987463335903157713e-01 3.22142321349414423314883038074186e-02 2.66226894560845240428337277194260e-03
            1.27799603324881029531301379620345e-02 1.31379538178000160320024747306888e-01 2.92222801419447005683016627131110e-03
            1.31379538178000160320024747306888e-01 8.55840501489511740196292066684691e-01 2.92222801419447005683016627131110e-03
            8.55840501489511740196292066684691e-01 1.27799603324881029531301379620345e-02 2.92222801419447005683016627131110e-03
            1.95430723381915305569833662957535e-01 1.40953400791673909980827872345799e-02 3.65400299007099822051758408747446e-03
            1.40953400791673909980827872345799e-02 7.90473936538917398841874728532275e-01 3.65400299007099822051758408747446e-03
            7.90473936538917398841874728532275e-01 1.95430723381915305569833662957535e-01 3.65400299007099822051758408747446e-03
            9.77669169905016482990944837183633e-02 3.50007516628948259218567784500920e-02 3.95316767664596370457363860850819e-03
            3.50007516628948259218567784500920e-02 8.67232331346603491084579218295403e-01 3.95316767664596370457363860850819e-03
            8.67232331346603491084579218295403e-01 9.77669169905016482990944837183633e-02 3.95316767664596370457363860850819e-03
            1.66006628441778165961029145591965e-02 3.38676752265837932576175717258593e-01 4.82797893544456221959215724837122e-03
            3.38676752265837932576175717258593e-01 6.44722584889984240419380512321368e-01 4.82797893544456221959215724837122e-03
            6.44722584889984240419380512321368e-01 1.66006628441778165961029145591965e-02 4.82797893544456221959215724837122e-03
            4.13395082180740003341412602821947e-01 1.82025878269482105364929225288506e-02 5.15729450192981084960530679950352e-03
            1.82025878269482105364929225288506e-02 5.68402329992311883266609129350400e-01 5.15729450192981084960530679950352e-03
            5.68402329992311883266609129350400e-01 4.13395082180740003341412602821947e-01 5.15729450192981084960530679950352e-03
            6.83293746341590430004586664836097e-02 1.08675089954566678196457019112131e-01 6.10491927213127442974904823813631e-03
            1.08675089954566678196457019112131e-01 8.22995535411274348192023353476543e-01 6.10491927213127442974904823813631e-03
            8.22995535411274348192023353476543e-01 6.83293746341590430004586664836097e-02 6.10491927213127442974904823813631e-03
            3.92153078432324539437381361040025e-02 2.14338430054550499459509182997863e-01 6.41934116104962010362910973526596e-03
            2.14338430054550499459509182997863e-01 7.46446262102216984146707545733079e-01 6.41934116104962010362910973526596e-03
            7.46446262102216984146707545733079e-01 3.92153078432324539437381361040025e-02 6.41934116104962010362910973526596e-03
            2.82851404387685689822973245100002e-01 4.32505997246091511421184350183466e-02 7.33339884886187540896651526622918e-03
            4.32505997246091511421184350183466e-02 6.73897995887705159034908319881652e-01 7.33339884886187540896651526622918e-03
            6.73897995887705159034908319881652e-01 2.82851404387685689822973245100002e-01 7.33339884886187540896651526622918e-03
            1.63149078355705329235192380110675e-01 7.46324203435316663179932561433816e-02 7.70252693118283777301558146177740e-03
            7.46324203435316663179932561433816e-02 7.62218501300762962813450940302573e-01 7.70252693118283777301558146177740e-03
            7.62218501300762962813450940302573e-01 1.63149078355705329235192380110675e-01 7.70252693118283777301558146177740e-03
            4.36290583906621953769189303784515e-01 5.16537077780689246608858411491383e-01 8.28779693743914719350485142967955e-03
            5.16537077780689246608858411491383e-01 4.71723383126887441108010534662753e-02 8.28779693743914719350485142967955e-03
            4.71723383126887441108010534662753e-02 4.36290583906621953769189303784515e-01 8.28779693743914719350485142967955e-03
            1.23069087330845969296966302408691e-01 1.73946464302243086752497447378119e-01 9.99461019059759599170167376769314e-03
            1.73946464302243086752497447378119e-01 7.02984448366910985583899673656560e-01 9.99461019059759599170167376769314e-03
            7.02984448366910985583899673656560e-01 1.23069087330845969296966302408691e-01 9.99461019059759599170167376769314e-03
            8.32579050810742932187125120435667e-02 2.99185227797303610053347711073002e-01 1.01487087387537137928017472177089e-02
            2.99185227797303610053347711073002e-01 6.17556867121622055094576353440061e-01 1.01487087387537137928017472177089e-02
            6.17556867121622055094576353440061e-01 8.32579050810742932187125120435667e-02 1.01487087387537137928017472177089e-02
            3.76028943583988584098420915324823e-01 8.64379444693639392705009072415123e-02 1.08183452528902452477455042867405e-02
            8.64379444693639392705009072415123e-02 5.37533111946647546020017216505948e-01 1.08183452528902452477455042867405e-02
            5.37533111946647546020017216505948e-01 3.76028943583988584098420915324823e-01 1.08183452528902452477455042867405e-02
            2.39866290944955318220621620639577e-01 1.30115610897138883617074611720454e-01 1.15127386814758876643072227352604e-02
            1.30115610897138883617074611720454e-01 6.30018098157905770406728152011055e-01 1.15127386814758876643072227352604e-02
            6.30018098157905770406728152011055e-01 2.39866290944955318220621620639577e-01 1.15127386814758876643072227352604e-02
            1.37221148823993804866105961082212e-01 3.91013201302957691396500194969121e-01 1.31617468738862725824434463106627e-02
            3.91013201302957691396500194969121e-01 4.71765649873048475981818228319753e-01 1.31617468738862725824434463106627e-02
            4.71765649873048475981818228319753e-01 1.37221148823993804866105961082212e-01 1.31617468738862725824434463106627e-02
            1.89932130608380905778531655414554e-01 2.50321346008893164114539331421838e-01 1.35240488135174967421203717776734e-02
            2.50321346008893164114539331421838e-01 5.59746523382725902351353397534695e-01 1.35240488135174967421203717776734e-02
            5.59746523382725902351353397534695e-01 1.89932130608380905778531655414554e-01 1.35240488135174967421203717776734e-02
            3.24158256395638832980665711147594e-01 1.94724488881111906790621901564009e-01 1.44971698648032557704690148625559e-02
            1.94724488881111906790621901564009e-01 4.81117254723249176961985540401656e-01 1.44971698648032557704690148625559e-02
            4.81117254723249176961985540401656e-01 3.24158256395638832980665711147594e-01 1.44971698648032557704690148625559e-02
            3.30862912147051713152023921793443e-01 4.07445324376853323933289630076615e-01 1.56953608107867174925598163781615e-02
            4.07445324376853323933289630076615e-01 2.61691763476095018425837679387769e-01 1.56953608107867174925598163781615e-02
            2.61691763476095018425837679387769e-01 3.30862912147051713152023921793443e-01 1.56953608107867174925598163781615e-02
            ];
        
    case  21
        
        % ALG. DEG.:   21
        % PTS CARD.:   85
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.6e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 7.96362196085264456568708624217834e-03
            1.08833811258291848766965870254353e-02 1.08437360200908869400926448633982e-02 7.70012405206899541373899431562222e-04
            1.08437360200908869400926448633982e-02 9.78272882854079872672059536853340e-01 7.70012405206899541373899431562222e-04
            9.78272882854079872672059536853340e-01 1.08833811258291848766965870254353e-02 7.70012405206899541373899431562222e-04
            1.34483082013301280532857617799891e-01 5.89401791446114188544047607365428e-03 1.48201910561772138424929678990338e-03
            5.89401791446114188544047607365428e-03 8.59622900072237627888682709453860e-01 1.48201910561772138424929678990338e-03
            8.59622900072237627888682709453860e-01 1.34483082013301280532857617799891e-01 1.48201910561772138424929678990338e-03
            5.60697914456082802114877949861693e-02 9.34072029515549240308303069468820e-01 1.57128523968988551701153610906658e-03
            9.34072029515549240308303069468820e-01 9.85817903884245172463351991609670e-03 1.57128523968988551701153610906658e-03
            9.85817903884245172463351991609670e-03 5.60697914456082802114877949861693e-02 1.57128523968988551701153610906658e-03
            5.62307971638516188028766862316843e-02 1.04111689007461219458328116616030e-02 1.66127886475559132656298899632930e-03
            1.04111689007461219458328116616030e-02 9.33358033935402287006866117735626e-01 1.66127886475559132656298899632930e-03
            9.33358033935402287006866117735626e-01 5.62307971638516188028766862316843e-02 1.66127886475559132656298899632930e-03
            4.95492250545260581873208138858899e-01 5.30139353929082006444684083135144e-03 2.03547106383759847034187195902177e-03
            5.30139353929082006444684083135144e-03 4.99206355915448596327621544332942e-01 2.03547106383759847034187195902177e-03
            4.99206355915448596327621544332942e-01 4.95492250545260581873208138858899e-01 2.03547106383759847034187195902177e-03
            1.33413544221436503045552512958238e-01 8.56502202113258115367955269903177e-01 2.35329019457066780651732784690466e-03
            8.56502202113258115367955269903177e-01 1.00842536653054093420678327674977e-02 2.35329019457066780651732784690466e-03
            1.00842536653054093420678327674977e-02 1.33413544221436503045552512958238e-01 2.35329019457066780651732784690466e-03
            2.37788853273795369736021143580729e-01 7.53553830034851435115683671028819e-01 2.59294011981360756138870193865387e-03
            7.53553830034851435115683671028819e-01 8.65731669135316739271956976153888e-03 2.59294011981360756138870193865387e-03
            8.65731669135316739271956976153888e-03 2.37788853273795369736021143580729e-01 2.59294011981360756138870193865387e-03
            2.36675822687569759050774109709891e-01 1.04315998683223633974881394692602e-02 3.02658613647318426240206967747781e-03
            1.04315998683223633974881394692602e-02 7.52892577444107868878120370936813e-01 3.02658613647318426240206967747781e-03
            7.52892577444107868878120370936813e-01 2.36675822687569759050774109709891e-01 3.02658613647318426240206967747781e-03
            3.59237431543587948379325780479121e-01 9.77441311908519919193771841037233e-03 3.24868657739030199799179499109414e-03
            9.77441311908519919193771841037233e-03 6.30988155337326772631456606177380e-01 3.24868657739030199799179499109414e-03
            6.30988155337326772631456606177380e-01 3.59237431543587948379325780479121e-01 3.24868657739030199799179499109414e-03
            3.62123795542131587943401882512262e-01 6.27482731394552795123331634385977e-01 3.48666208271716836566977981703985e-03
            6.27482731394552795123331634385977e-01 1.03934730633156169332664831017610e-02 3.48666208271716836566977981703985e-03
            1.03934730633156169332664831017610e-02 3.62123795542131587943401882512262e-01 3.48666208271716836566977981703985e-03
            5.47064625266307918982633395899029e-02 8.93376258603483019982149926363491e-01 3.60293779467318669648689954954079e-03
            8.93376258603483019982149926363491e-01 5.19172788698861742417989262321498e-02 3.60293779467318669648689954954079e-03
            5.19172788698861742417989262321498e-02 5.47064625266307918982633395899029e-02 3.60293779467318669648689954954079e-03
            1.28497845564646773208750119010801e-01 3.68128421478705475378134792663332e-02 4.18250641443536613078846286839507e-03
            3.68128421478705475378134792663332e-02 8.34689312287482665375648593908409e-01 4.18250641443536613078846286839507e-03
            8.34689312287482665375648593908409e-01 1.28497845564646773208750119010801e-01 4.18250641443536613078846286839507e-03
            1.32045621485988418086776619020384e-01 8.16750542291406311790069594280794e-01 5.11329816841889044087832161267215e-03
            8.16750542291406311790069594280794e-01 5.12038362226052701231537866988219e-02 5.11329816841889044087832161267215e-03
            5.12038362226052701231537866988219e-02 1.32045621485988418086776619020384e-01 5.11329816841889044087832161267215e-03
            4.80002887465515781784120008524042e-01 3.46233167655783727334473098835588e-02 5.87046436829854155481367428137673e-03
            3.46233167655783727334473098835588e-02 4.85373795768905824665750969870714e-01 5.87046436829854155481367428137673e-03
            4.85373795768905824665750969870714e-01 4.80002887465515781784120008524042e-01 5.87046436829854155481367428137673e-03
            2.35698652486183879428693899171776e-01 7.17639780294608065069894564658171e-01 6.09162472206678431291271280656474e-03
            7.17639780294608065069894564658171e-01 4.66615672192079999902603049122263e-02 6.09162472206678431291271280656474e-03
            4.66615672192079999902603049122263e-02 2.35698652486183879428693899171776e-01 6.09162472206678431291271280656474e-03
            1.21235453505120910344672324754356e-01 9.87172892132079649041287439104053e-02 6.43274473743770617423232138776257e-03
            9.87172892132079649041287439104053e-02 7.80047257281671124751198931335239e-01 6.43274473743770617423232138776257e-03
            7.80047257281671124751198931335239e-01 1.21235453505120910344672324754356e-01 6.43274473743770617423232138776257e-03
            2.26921312534569846741661081068742e-01 5.33826956732506574532237664243439e-02 6.54764913100841332294077545839173e-03
            5.33826956732506574532237664243439e-02 7.19695991792179468049539536878001e-01 6.54764913100841332294077545839173e-03
            7.19695991792179468049539536878001e-01 2.26921312534569846741661081068742e-01 6.54764913100841332294077545839173e-03
            3.52725956002229834584937862018705e-01 5.93956020138616369585804477537749e-01 7.04335952667112241698532670852728e-03
            5.93956020138616369585804477537749e-01 5.33180238591538513404088917013723e-02 7.04335952667112241698532670852728e-03
            5.33180238591538513404088917013723e-02 3.52725956002229834584937862018705e-01 7.04335952667112241698532670852728e-03
            3.46394760980635263436511195322964e-01 5.11343671910869101981944595536334e-02 7.09201535978721143288483474975692e-03
            5.11343671910869101981944595536334e-02 6.02470871828277854120869960752316e-01 7.09201535978721143288483474975692e-03
            6.02470871828277854120869960752316e-01 3.46394760980635263436511195322964e-01 7.09201535978721143288483474975692e-03
            2.08545376179994440546394685043197e-01 1.26661874088773246382899628770247e-01 8.61041578380116606750149799154315e-03
            1.26661874088773246382899628770247e-01 6.64792749731232257559554454928730e-01 8.61041578380116606750149799154315e-03
            6.64792749731232257559554454928730e-01 2.08545376179994440546394685043197e-01 8.61041578380116606750149799154315e-03
            1.16641181787498937616476268885890e-01 1.93583513707742599363470503703866e-01 8.70681197967806269621959813775902e-03
            1.93583513707742599363470503703866e-01 6.89775304504758435264477611781331e-01 8.70681197967806269621959813775902e-03
            6.89775304504758435264477611781331e-01 1.16641181787498937616476268885890e-01 8.70681197967806269621959813775902e-03
            3.01681731659511120913208515048609e-01 5.79485561285726924474204224679852e-01 9.04183777182423521756593487452847e-03
            5.79485561285726924474204224679852e-01 1.18832707054762010123738491529366e-01 9.04183777182423521756593487452847e-03
            1.18832707054762010123738491529366e-01 3.01681731659511120913208515048609e-01 9.04183777182423521756593487452847e-03
            3.59766505094364252848038177035050e-01 4.65449578776416461600717866531340e-01 9.57456938333661397577678542347712e-03
            4.65449578776416461600717866531340e-01 1.74783916129219285551243956433609e-01 9.57456938333661397577678542347712e-03
            1.74783916129219285551243956433609e-01 3.59766505094364252848038177035050e-01 9.57456938333661397577678542347712e-03
            3.17727894779720854145921293820720e-01 1.21922765864560744830669136717916e-01 9.61390963530625239363835987660423e-03
            1.21922765864560744830669136717916e-01 5.60349339355718401023409569461364e-01 9.61390963530625239363835987660423e-03
            5.60349339355718401023409569461364e-01 3.17727894779720854145921293820720e-01 9.61390963530625239363835987660423e-03
            4.50599237708166233673523493052926e-01 9.80260802306452266874359224857471e-02 1.00158383807107658619584000803115e-02
            9.80260802306452266874359224857471e-02 4.51374682061188525761252776646870e-01 1.00158383807107658619584000803115e-02
            4.51374682061188525761252776646870e-01 4.50599237708166233673523493052926e-01 1.00158383807107658619584000803115e-02
            2.66780208878214575296539123883122e-01 3.09304671639405781569109876727453e-01 1.03825614307465671076746716039452e-02
            3.09304671639405781569109876727453e-01 4.23915119482379698645502230647253e-01 1.03825614307465671076746716039452e-02
            4.23915119482379698645502230647253e-01 2.66780208878214575296539123883122e-01 1.03825614307465671076746716039452e-02
            2.14913322085512142889385245325684e-01 2.23615271231608347957120486171334e-01 1.19075263531945689521318598735888e-02
            2.23615271231608347957120486171334e-01 5.61471406682879425886767421616241e-01 1.19075263531945689521318598735888e-02
            5.61471406682879425886767421616241e-01 2.14913322085512142889385245325684e-01 1.19075263531945689521318598735888e-02
            3.41820828914985830326855875682668e-01 2.03059575102732176032915845098614e-01 1.19538232815810784720467196962090e-02
            2.03059575102732176032915845098614e-01 4.55119595982282021395803894847631e-01 1.19538232815810784720467196962090e-02
            4.55119595982282021395803894847631e-01 3.41820828914985830326855875682668e-01 1.19538232815810784720467196962090e-02
            ];
        
    case  22
        
        % ALG. DEG.:   22
        % PTS CARD.:   93
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 7.6e-16
        
        xyw=[
            1.17033848378872841304587648636470e-03 4.78536918807590805347018658721936e-02 4.87706020469868661414841737311576e-04
            4.78536918807590805347018658721936e-02 9.50975969635452167416644897457445e-01 4.87706020469868661414841737311576e-04
            9.50975969635452167416644897457445e-01 1.17033848378872841304587648636470e-03 4.87706020469868661414841737311576e-04
            1.09943655740260652631246074406590e-02 9.10867175372020998480593334534205e-03 6.55222610574895123272676222825339e-04
            9.10867175372020998480593334534205e-03 9.79896962672253679649259083817014e-01 6.55222610574895123272676222825339e-04
            9.79896962672253679649259083817014e-01 1.09943655740260652631246074406590e-02 6.55222610574895123272676222825339e-04
            1.06816925225288752131225233910072e-03 2.02709063387925630950192612544924e-01 9.09692042324600669452039358731099e-04
            2.02709063387925630950192612544924e-01 7.96222767359821559374211119575193e-01 9.09692042324600669452039358731099e-04
            7.96222767359821559374211119575193e-01 1.06816925225288752131225233910072e-03 9.09692042324600669452039358731099e-04
            2.21228350592110340616658348267265e-03 4.21769070671465406974220968550071e-01 1.24269485894917082122179685654828e-03
            4.21769070671465406974220968550071e-01 5.76018645822613528650890657445416e-01 1.24269485894917082122179685654828e-03
            5.76018645822613528650890657445416e-01 2.21228350592110340616658348267265e-03 1.24269485894917082122179685654828e-03
            5.72025987162534596541796361179877e-02 8.56087099920958480292476622253162e-03 1.39715288121358824011131183340240e-03
            8.56087099920958480292476622253162e-03 9.34236530284537014523493780870922e-01 1.39715288121358824011131183340240e-03
            9.34236530284537014523493780870922e-01 5.72025987162534596541796361179877e-02 1.39715288121358824011131183340240e-03
            3.53024211615276806153218558392837e-01 6.09424125055589051280779955277467e-03 1.77407428600574994942418172172438e-03
            6.09424125055589051280779955277467e-03 6.40881547134167228740864175051684e-01 1.77407428600574994942418172172438e-03
            6.40881547134167228740864175051684e-01 3.53024211615276806153218558392837e-01 1.77407428600574994942418172172438e-03
            1.36788957839153818563104891836701e-01 7.84998062848008687497625146534119e-03 1.88931332815918764413132269197604e-03
            7.84998062848008687497625146534119e-03 8.55361061532366129256388376234099e-01 1.88931332815918764413132269197604e-03
            8.55361061532366129256388376234099e-01 1.36788957839153818563104891836701e-01 1.88931332815918764413132269197604e-03
            2.41209563760037715152151349684573e-01 7.02726355172032801643666743984795e-03 2.01913314251987198455484673331739e-03
            7.02726355172032801643666743984795e-03 7.51763172688241887442472943803295e-01 2.01913314251987198455484673331739e-03
            7.51763172688241887442472943803295e-01 2.41209563760037715152151349684573e-01 2.01913314251987198455484673331739e-03
            1.16737477866878831822550210972622e-02 1.12641216429535215293178396223084e-01 2.31660679651168400008942604983986e-03
            1.12641216429535215293178396223084e-01 8.75685035783776988260740381520009e-01 2.31660679651168400008942604983986e-03
            8.75685035783776988260740381520009e-01 1.16737477866878831822550210972622e-02 2.31660679651168400008942604983986e-03
            3.01148996347992969901952164946124e-02 4.60558965203824444145652705628891e-02 2.46536317015107241459870301980573e-03
            4.60558965203824444145652705628891e-02 9.23829203844818258595239512942499e-01 2.46536317015107241459870301980573e-03
            9.23829203844818258595239512942499e-01 3.01148996347992969901952164946124e-02 2.46536317015107241459870301980573e-03
            4.56500154387454060156414925586432e-01 1.14379077070311217789733149174936e-02 3.20785199538873395594396065177989e-03
            1.14379077070311217789733149174936e-02 5.32061937905514814595164807542460e-01 3.20785199538873395594396065177989e-03
            5.32061937905514814595164807542460e-01 4.56500154387454060156414925586432e-01 3.20785199538873395594396065177989e-03
            1.18075946189018649024049523177382e-02 3.04996921690560307549588969777687e-01 3.32421252207346388726771913013636e-03
            3.04996921690560307549588969777687e-01 6.83195483690537841425793885719031e-01 3.32421252207346388726771913013636e-03
            6.83195483690537841425793885719031e-01 1.18075946189018649024049523177382e-02 3.32421252207346388726771913013636e-03
            9.61505587301470371341949316956743e-02 4.26352318690261661293661177296599e-02 3.96867722784416832731979596360361e-03
            4.26352318690261661293661177296599e-02 8.61214209400826824492014566203579e-01 3.96867722784416832731979596360361e-03
            8.61214209400826824492014566203579e-01 9.61505587301470371341949316956743e-02 3.96867722784416832731979596360361e-03
            3.06376574992079410419965057599256e-02 1.96990580025709072220507778183674e-01 4.69591283560160614035661197362970e-03
            1.96990580025709072220507778183674e-01 7.72371762475082990206942668010015e-01 4.69591283560160614035661197362970e-03
            7.72371762475082990206942668010015e-01 3.06376574992079410419965057599256e-02 4.69591283560160614035661197362970e-03
            1.94815244430806083775209458508471e-01 3.86641674378723973481974951482698e-02 4.88329231189547560698649419919093e-03
            3.86641674378723973481974951482698e-02 7.66520588131321511937699142436031e-01 4.88329231189547560698649419919093e-03
            7.66520588131321511937699142436031e-01 1.94815244430806083775209458508471e-01 4.88329231189547560698649419919093e-03
            6.03655030702410985710137936166575e-02 1.06610220689147924821860158317577e-01 4.96965611820962896438835798562650e-03
            1.06610220689147924821860158317577e-01 8.33024276240610928034868720715167e-01 4.96965611820962896438835798562650e-03
            8.33024276240610928034868720715167e-01 6.03655030702410985710137936166575e-02 4.96965611820962896438835798562650e-03
            3.15090899482059372260778218333144e-01 3.60898201743391419671347364328540e-02 5.39285897517053670890962990824846e-03
            3.60898201743391419671347364328540e-02 6.48819280343601478833193141326774e-01 5.39285897517053670890962990824846e-03
            6.48819280343601478833193141326774e-01 3.15090899482059372260778218333144e-01 5.39285897517053670890962990824846e-03
            4.11358367215817199991079178289510e-01 5.56390353961855654851831332052825e-01 5.59935690856001519466467186703085e-03
            5.56390353961855654851831332052825e-01 3.22512788223271451570894896576647e-02 5.59935690856001519466467186703085e-03
            3.22512788223271451570894896576647e-02 4.11358367215817199991079178289510e-01 5.59935690856001519466467186703085e-03
            1.44020350630489235577158524392871e-01 9.77155901845127572435956153640291e-02 6.79622462533692028169296506234787e-03
            9.77155901845127572435956153640291e-02 7.58264059184998062690397091500927e-01 6.79622462533692028169296506234787e-03
            7.58264059184998062690397091500927e-01 1.44020350630489235577158524392871e-01 6.79622462533692028169296506234787e-03
            4.20757709102121901789672619997873e-01 5.55288374894526703950958790301229e-02 7.05364365658125037211734209563474e-03
            5.55288374894526703950958790301229e-02 5.23713453408425344548504654085264e-01 7.05364365658125037211734209563474e-03
            5.23713453408425344548504654085264e-01 4.20757709102121901789672619997873e-01 7.05364365658125037211734209563474e-03
            6.09498995807896612464382712914812e-02 2.92056016141072904979125723912148e-01 7.17274949295997488879494952129789e-03
            2.92056016141072904979125723912148e-01 6.46994084278137426835542100889143e-01 7.17274949295997488879494952129789e-03
            6.46994084278137426835542100889143e-01 6.09498995807896612464382712914812e-02 7.17274949295997488879494952129789e-03
            9.69351064455874905645416106381163e-02 1.84224156891908047439798679079104e-01 7.51657234055716320930695673041555e-03
            1.84224156891908047439798679079104e-01 7.18840736662504475873447518097237e-01 7.51657234055716320930695673041555e-03
            7.18840736662504475873447518097237e-01 9.69351064455874905645416106381163e-02 7.51657234055716320930695673041555e-03
            2.53783465620646964122641975336592e-01 9.22286849014402332347373203447205e-02 7.91883890190281380938586153206415e-03
            9.22286849014402332347373203447205e-02 6.53987849477912774887045088689774e-01 7.91883890190281380938586153206415e-03
            6.53987849477912774887045088689774e-01 2.53783465620646964122641975336592e-01 7.91883890190281380938586153206415e-03
            3.48867101402162760326319812520524e-01 1.16235552070709735605902324095950e-01 8.55725277413337676035709478128410e-03
            1.16235552070709735605902324095950e-01 5.34897346527127615090080325899180e-01 8.55725277413337676035709478128410e-03
            5.34897346527127615090080325899180e-01 3.48867101402162760326319812520524e-01 8.55725277413337676035709478128410e-03
            1.90581558653074217835055037539860e-01 1.72148246666744941801852064600098e-01 8.98564555031942067964134679414201e-03
            1.72148246666744941801852064600098e-01 6.37270194680180868118668513488956e-01 8.98564555031942067964134679414201e-03
            6.37270194680180868118668513488956e-01 1.90581558653074217835055037539860e-01 8.98564555031942067964134679414201e-03
            9.69881143348128027525945071829483e-02 3.96552131508626504174941374003538e-01 9.27251659845123449299464368777990e-03
            3.96552131508626504174941374003538e-01 5.06459754156560748583615350071341e-01 9.27251659845123449299464368777990e-03
            5.06459754156560748583615350071341e-01 9.69881143348128027525945071829483e-02 9.27251659845123449299464368777990e-03
            1.41266334630631706925640855843085e-01 2.75379650245183016377126250517904e-01 9.69905562317903077729219774028024e-03
            2.75379650245183016377126250517904e-01 5.83354015124185165674930431123357e-01 9.69905562317903077729219774028024e-03
            5.83354015124185165674930431123357e-01 1.41266334630631706925640855843085e-01 9.69905562317903077729219774028024e-03
            4.12018101979952200419887731186463e-01 1.68573884290088321646905455963861e-01 9.76183529320075356328079152490318e-03
            1.68573884290088321646905455963861e-01 4.19408013729959394666479965962935e-01 9.76183529320075356328079152490318e-03
            4.19408013729959394666479965962935e-01 4.12018101979952200419887731186463e-01 9.76183529320075356328079152490318e-03
            2.76209339065321413109188597445609e-01 1.99199356373184033319745367407450e-01 1.03127659848185199154002589239099e-02
            1.99199356373184033319745367407450e-01 5.24591304561494498059914803889114e-01 1.03127659848185199154002589239099e-02
            5.24591304561494498059914803889114e-01 2.76209339065321413109188597445609e-01 1.03127659848185199154002589239099e-02
            2.18610614672497399757489233707020e-01 3.05685651047730433393923021867522e-01 1.06569112100914267088347386902569e-02
            3.05685651047730433393923021867522e-01 4.75703734279772139093012128796545e-01 1.06569112100914267088347386902569e-02
            4.75703734279772139093012128796545e-01 2.18610614672497399757489233707020e-01 1.06569112100914267088347386902569e-02
            3.36799046311846372869780452674604e-01 2.69814345781991038464298071630765e-01 1.17638765835114608715450401632552e-02
            2.69814345781991038464298071630765e-01 3.93386607906162533154770244436804e-01 1.17638765835114608715450401632552e-02
            3.93386607906162533154770244436804e-01 3.36799046311846372869780452674604e-01 1.17638765835114608715450401632552e-02
            ];
        
    case  23
        
        % ALG. DEG.:   23
        % PTS CARD.:  100
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.7e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 1.25694357301468478266270878407340e-02
            9.41394903300460904327362499088849e-03 8.64644445189447809207994311009315e-03 5.33058841253013855794595787784829e-04
            8.64644445189447809207994311009315e-03 9.81939606515100926742434239713475e-01 5.33058841253013855794595787784829e-04
            9.81939606515100926742434239713475e-01 9.41394903300460904327362499088849e-03 5.33058841253013855794595787784829e-04
            4.82410218908890808548611817130336e-04 1.08821420580560865309927010002866e-01 5.54936575277445703208156757568759e-04
            1.08821420580560865309927010002866e-01 8.90696169200530185605657607084140e-01 5.54936575277445703208156757568759e-04
            8.90696169200530185605657607084140e-01 4.82410218908890808548611817130336e-04 5.54936575277445703208156757568759e-04
            4.96335565629436092915938161240774e-01 5.03294735930981729232769339432707e-01 8.14811977822814392791173077057465e-04
            5.03294735930981729232769339432707e-01 3.69698439582122340141268068691716e-04 8.14811977822814392791173077057465e-04
            3.69698439582122340141268068691716e-04 4.96335565629436092915938161240774e-01 8.14811977822814392791173077057465e-04
            8.61452759024325040448921697588958e-03 4.49648589409331542188219543731975e-02 1.10264584656888744475244923393120e-03
            4.49648589409331542188219543731975e-02 9.46420613468823557212772357161157e-01 1.10264584656888744475244923393120e-03
            9.46420613468823557212772357161157e-01 8.61452759024325040448921697588958e-03 1.10264584656888744475244923393120e-03
            4.91006634687758056201367651283363e-02 8.24249708242969815474854300418883e-03 1.15877715989831956908129306782484e-03
            8.24249708242969815474854300418883e-03 9.42656839448794414693111320957541e-01 1.15877715989831956908129306782484e-03
            9.42656839448794414693111320957541e-01 4.91006634687758056201367651283363e-02 1.15877715989831956908129306782484e-03
            2.66041607735387425695550156490299e-03 2.94704482021720892692684401481529e-01 1.16081935080770103085878730553304e-03
            2.94704482021720892692684401481529e-01 7.02635101900925151952037595037837e-01 1.16081935080770103085878730553304e-03
            7.02635101900925151952037595037837e-01 2.66041607735387425695550156490299e-03 1.16081935080770103085878730553304e-03
            3.30984469269412107106376197407371e-01 3.82066849179892769131083873901389e-03 1.50183075569778300140111326754777e-03
            3.82066849179892769131083873901389e-03 6.65194862238788875430373082053848e-01 1.50183075569778300140111326754777e-03
            6.65194862238788875430373082053848e-01 3.30984469269412107106376197407371e-01 1.50183075569778300140111326754777e-03
            1.18154505921019392644666368141770e-01 8.41315741466065573905819263700323e-03 1.77381096176873259348538702795395e-03
            8.41315741466065573905819263700323e-03 8.73432336664319963759339771058876e-01 1.77381096176873259348538702795395e-03
            8.73432336664319963759339771058876e-01 1.18154505921019392644666368141770e-01 1.77381096176873259348538702795395e-03
            2.14073097879028895196285020574578e-01 7.20539757650225185653392045992405e-03 1.90132581827351246697710429600647e-03
            7.20539757650225185653392045992405e-03 7.78721504544468867692330604768358e-01 1.90132581827351246697710429600647e-03
            7.78721504544468867692330604768358e-01 2.14073097879028895196285020574578e-01 1.90132581827351246697710429600647e-03
            4.45774128773336539288685287374392e-02 4.30263570230853334419585110026674e-02 2.36716235132847186914406556468293e-03
            4.30263570230853334419585110026674e-02 9.12396230099581040384748575888807e-01 2.36716235132847186914406556468293e-03
            9.12396230099581040384748575888807e-01 4.45774128773336539288685287374392e-02 2.36716235132847186914406556468293e-03
            1.91431395153490890859160344916745e-01 3.11692698898602960078108026209520e-02 2.39102276472424142214623365987336e-03
            3.11692698898602960078108026209520e-02 7.77399334956648813133028852462303e-01 2.39102276472424142214623365987336e-03
            7.77399334956648813133028852462303e-01 1.91431395153490890859160344916745e-01 2.39102276472424142214623365987336e-03
            1.05850984567690802745643097182437e-02 1.91864903048277385400055550235265e-01 2.52746008365003528572945157293361e-03
            1.91864903048277385400055550235265e-01 7.97549998494953560346232279698597e-01 2.52746008365003528572945157293361e-03
            7.97549998494953560346232279698597e-01 1.05850984567690802745643097182437e-02 2.52746008365003528572945157293361e-03
            2.54202264290056406670359478994214e-02 1.05407606480096824319581116924383e-01 3.02260306724195853753101026484273e-03
            1.05407606480096824319581116924383e-01 8.69172167090897507257807319547283e-01 3.02260306724195853753101026484273e-03
            8.69172167090897507257807319547283e-01 2.54202264290056406670359478994214e-02 3.02260306724195853753101026484273e-03
            5.91292919761095991937338567367988e-01 1.21072892452521504308426969487300e-02 3.15635122284491521221783116857296e-03
            1.21072892452521504308426969487300e-02 3.96599790993651835080413547984790e-01 3.15635122284491521221783116857296e-03
            3.96599790993651835080413547984790e-01 5.91292919761095991937338567367988e-01 3.15635122284491521221783116857296e-03
            1.04903656520941024332671531738015e-01 4.33709753919495621943269725306891e-02 3.38820274356900008952409031337538e-03
            4.33709753919495621943269725306891e-02 8.51725368087109413473001495731296e-01 3.38820274356900008952409031337538e-03
            8.51725368087109413473001495731296e-01 1.04903656520941024332671531738015e-01 3.38820274356900008952409031337538e-03
            4.27217446018734792190230109554250e-01 1.66970788554706185691234310297659e-02 3.94239621381627623752930134060080e-03
            1.66970788554706185691234310297659e-02 5.56085475125794537198942180111771e-01 3.94239621381627623752930134060080e-03
            5.56085475125794537198942180111771e-01 4.27217446018734792190230109554250e-01 3.94239621381627623752930134060080e-03
            1.77899296750879898354469332844019e-01 6.45110680237001715697786607961461e-02 4.36845382037373986677808446188465e-03
            6.45110680237001715697786607961461e-02 7.57589635225419888442388582916465e-01 4.36845382037373986677808446188465e-03
            7.57589635225419888442388582916465e-01 1.77899296750879898354469332844019e-01 4.36845382037373986677808446188465e-03
            8.05482155825715390173868968304305e-02 1.01976993969515816740312175170402e-01 4.71720292699666973040839579311978e-03
            1.01976993969515816740312175170402e-01 8.17474790447912713631239967071451e-01 4.71720292699666973040839579311978e-03
            8.17474790447912713631239967071451e-01 8.05482155825715390173868968304305e-02 4.71720292699666973040839579311978e-03
            3.13173112798550493107718750707136e-02 2.85342978045655892849197243776871e-01 4.74543971390964033907344798990380e-03
            2.85342978045655892849197243776871e-01 6.83339710674489064778924785059644e-01 4.74543971390964033907344798990380e-03
            6.83339710674489064778924785059644e-01 3.13173112798550493107718750707136e-02 4.74543971390964033907344798990380e-03
            2.95446758742554271837121859789477e-01 3.28810125354330601288310731433739e-02 5.10548132810887539401178969455941e-03
            3.28810125354330601288310731433739e-02 6.71672228722012709667410490510520e-01 5.10548132810887539401178969455941e-03
            6.71672228722012709667410490510520e-01 2.95446758742554271837121859789477e-01 5.10548132810887539401178969455941e-03
            5.41132173539479316426437094378343e-02 1.83859582702943807897000283446687e-01 5.25962666774252175089232608229395e-03
            1.83859582702943807897000283446687e-01 7.62027199943108302093719430558849e-01 5.25962666774252175089232608229395e-03
            7.62027199943108302093719430558849e-01 5.41132173539479316426437094378343e-02 5.25962666774252175089232608229395e-03
            4.27400654280490135961123598917766e-02 4.43558986603905025525307337375125e-01 5.84456018763875907728877479030416e-03
            4.43558986603905025525307337375125e-01 5.13700947968045995573049822269240e-01 5.84456018763875907728877479030416e-03
            5.13700947968045995573049822269240e-01 4.27400654280490135961123598917766e-02 5.84456018763875907728877479030416e-03
            1.45187675973377866567304295131180e-01 1.29795450451007293724003943680145e-01 6.85020013970072307735170724640739e-03
            1.29795450451007293724003943680145e-01 7.25016873575614839708691761188675e-01 6.85020013970072307735170724640739e-03
            7.25016873575614839708691761188675e-01 1.45187675973377866567304295131180e-01 6.85020013970072307735170724640739e-03
            7.47139784787638483454230708957766e-02 3.29913961425314705699918249592884e-01 7.23022564954137799547728704396832e-03
            3.29913961425314705699918249592884e-01 5.95372060095921473710234295140253e-01 7.23022564954137799547728704396832e-03
            5.95372060095921473710234295140253e-01 7.47139784787638483454230708957766e-02 7.23022564954137799547728704396832e-03
            3.81968043130448131616816453970387e-01 6.73625275151304159226839374241536e-02 7.61569789179446145904739395859906e-03
            6.73625275151304159226839374241536e-02 5.50669429354421424704923992976546e-01 7.61569789179446145904739395859906e-03
            5.50669429354421424704923992976546e-01 3.81968043130448131616816453970387e-01 7.61569789179446145904739395859906e-03
            2.21783549821814751634718732020701e-01 6.68980984358149566304518884862773e-01 7.74293665972291839799668267119159e-03
            6.68980984358149566304518884862773e-01 1.09235465820035626549611151858699e-01 7.74293665972291839799668267119159e-03
            1.09235465820035626549611151858699e-01 2.21783549821814751634718732020701e-01 7.74293665972291839799668267119159e-03
            2.56870811513146835025622749526519e-01 9.60491211353747958989757194103731e-02 7.99447871827724948490256906552531e-03
            9.60491211353747958989757194103731e-02 6.47080067351478382953189338877564e-01 7.99447871827724948490256906552531e-03
            6.47080067351478382953189338877564e-01 2.56870811513146835025622749526519e-01 7.99447871827724948490256906552531e-03
            1.12751303426959964548714765442128e-01 4.23674182024581569283583348806133e-01 9.46451795294582336193212057651181e-03
            4.23674182024581569283583348806133e-01 4.63574514548458493923277501380653e-01 9.46451795294582336193212057651181e-03
            4.63574514548458493923277501380653e-01 1.12751303426959964548714765442128e-01 9.46451795294582336193212057651181e-03
            2.10323057441033522962925417232327e-01 1.87001109672332688216300766725908e-01 9.87404067836702981075802654231666e-03
            1.87001109672332688216300766725908e-01 6.02675832886633733309622584783938e-01 9.87404067836702981075802654231666e-03
            6.02675832886633733309622584783938e-01 2.10323057441033522962925417232327e-01 9.87404067836702981075802654231666e-03
            1.61575209179476625420690538703639e-01 2.99012948455699423266906933349674e-01 1.03338907798329795312275436458549e-02
            2.99012948455699423266906933349674e-01 5.39411842364823979067978143575601e-01 1.03338907798329795312275436458549e-02
            5.39411842364823979067978143575601e-01 1.61575209179476625420690538703639e-01 1.03338907798329795312275436458549e-02
            3.31656221355574820641720634739613e-01 1.50616235510018708909640849924472e-01 1.03991713323975526472864316929190e-02
            1.50616235510018708909640849924472e-01 5.17727543134406498204214130964829e-01 1.03991713323975526472864316929190e-02
            5.17727543134406498204214130964829e-01 3.31656221355574820641720634739613e-01 1.03991713323975526472864316929190e-02
            2.12294763316891205162662004113372e-01 3.84271485252512368102628670385457e-01 1.17877874203720679757712019863902e-02
            3.84271485252512368102628670385457e-01 4.03433751430596398979133709872258e-01 1.17877874203720679757712019863902e-02
            4.03433751430596398979133709872258e-01 2.12294763316891205162662004113372e-01 1.17877874203720679757712019863902e-02
            2.74113742189350695355187781387940e-01 2.58726723002178027321917852532351e-01 1.18459271543522540875992277165096e-02
            2.58726723002178027321917852532351e-01 4.67159534808471277322894366079709e-01 1.18459271543522540875992277165096e-02
            4.67159534808471277322894366079709e-01 2.74113742189350695355187781387940e-01 1.18459271543522540875992277165096e-02
            ];
        
    case  24
        
        % ALG. DEG.:   24
        % PTS CARD.:  109
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 7.8e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 5.34088846669340550771254427786516e-03
            7.27744218228642408829998089458968e-03 1.87178063161081678807129424768618e-03 1.71536069217019938697166292485008e-04
            1.87178063161081678807129424768618e-03 9.90850777186102704696679666085402e-01 1.71536069217019938697166292485008e-04
            9.90850777186102704696679666085402e-01 7.27744218228642408829998089458968e-03 1.71536069217019938697166292485008e-04
            9.37621417401595796947688654654485e-02 2.22584177630761334901188241985892e-03 6.11499851018703364027850888362536e-04
            2.22584177630761334901188241985892e-03 9.04012016483532798716282741224859e-01 6.11499851018703364027850888362536e-04
            9.04012016483532798716282741224859e-01 9.37621417401595796947688654654485e-02 6.11499851018703364027850888362536e-04
            2.75499567254815017069979887764930e-02 9.64825831627800223344593177898787e-01 8.19080136824596599653958151066035e-04
            9.64825831627800223344593177898787e-01 7.62421164671822637615150597412139e-03 8.19080136824596599653958151066035e-04
            7.62421164671822637615150597412139e-03 2.75499567254815017069979887764930e-02 8.19080136824596599653958151066035e-04
            3.88491565765760812900531107061397e-02 9.99221468468047303856938157196055e-03 1.11861297293506167446763832629131e-03
            9.99221468468047303856938157196055e-03 9.51158628738743461283888791513164e-01 1.11861297293506167446763832629131e-03
            9.51158628738743461283888791513164e-01 3.88491565765760812900531107061397e-02 1.11861297293506167446763832629131e-03
            8.45824643422994626451583144444157e-02 9.07583350374654984626943132752785e-01 1.42875093413913595123687638732690e-03
            9.07583350374654984626943132752785e-01 7.83418528304558048347416843171231e-03 1.42875093413913595123687638732690e-03
            7.83418528304558048347416843171231e-03 8.45824643422994626451583144444157e-02 1.42875093413913595123687638732690e-03
            1.69711974261767234439801654843905e-01 8.55094769491029395303005600226243e-03 1.84291480816372275908232580832191e-03
            8.55094769491029395303005600226243e-03 8.21737078043322521914149092481239e-01 1.84291480816372275908232580832191e-03
            8.21737078043322521914149092481239e-01 1.69711974261767234439801654843905e-01 1.84291480816372275908232580832191e-03
            2.61114259888929389763490007680957e-01 7.35523531115195392665606632931485e-03 1.84739772102621570108726967163193e-03
            7.35523531115195392665606632931485e-03 7.31530504799918657177215663978131e-01 1.84739772102621570108726967163193e-03
            7.31530504799918657177215663978131e-01 2.61114259888929389763490007680957e-01 1.84739772102621570108726967163193e-03
            1.67919705131083785154544329998316e-01 8.24005083539038452045133453793824e-01 1.95101600310397700621667382847590e-03
            8.24005083539038452045133453793824e-01 8.07521132987776280032221620786004e-03 1.95101600310397700621667382847590e-03
            8.07521132987776280032221620786004e-03 1.67919705131083785154544329998316e-01 1.95101600310397700621667382847590e-03
            3.66562445874964837333465084157069e-01 7.00566016949978002253951459010750e-03 2.01403454846550093032542250170991e-03
            7.00566016949978002253951459010750e-03 6.26431893955535445961402274406282e-01 2.01403454846550093032542250170991e-03
            6.26431893955535445961402274406282e-01 3.66562445874964837333465084157069e-01 2.01403454846550093032542250170991e-03
            2.71431359408183414672066646744497e-01 7.20628297746085721975362048397074e-01 2.26244036008083852090644327859081e-03
            7.20628297746085721975362048397074e-01 7.94034284573086335257130485842936e-03 2.26244036008083852090644327859081e-03
            7.94034284573086335257130485842936e-03 2.71431359408183414672066646744497e-01 2.26244036008083852090644327859081e-03
            4.83415122297192079159344757499639e-01 7.36119356684007599550545464239804e-03 2.26980146313370380675711324158783e-03
            7.36119356684007599550545464239804e-03 5.09223684135967857855575857684016e-01 2.26980146313370380675711324158783e-03
            5.09223684135967857855575857684016e-01 4.83415122297192079159344757499639e-01 2.26980146313370380675711324158783e-03
            3.87957374245691544167868869408267e-01 6.04329908418651196022608473867876e-01 2.37284772365591511630866783377769e-03
            6.04329908418651196022608473867876e-01 7.71271733565725980952265672385693e-03 2.37284772365591511630866783377769e-03
            7.71271733565725980952265672385693e-03 3.87957374245691544167868869408267e-01 2.37284772365591511630866783377769e-03
            5.20536116012078262094142644400563e-02 9.07672123610022807405073308473220e-01 2.59026828668722924886580472048081e-03
            9.07672123610022807405073308473220e-01 4.02742647887693872021941388084088e-02 2.59026828668722924886580472048081e-03
            4.02742647887693872021941388084088e-02 5.20536116012078262094142644400563e-02 2.59026828668722924886580472048081e-03
            2.29018909213725946338868766360974e-01 8.93452564981080527006795932720706e-02 2.65596161636161244221088217898341e-03
            8.93452564981080527006795932720706e-02 6.81635834288165987082663832552498e-01 2.65596161636161244221088217898341e-03
            6.81635834288165987082663832552498e-01 2.29018909213725946338868766360974e-01 2.65596161636161244221088217898341e-03
            9.49141955716360924411034716285940e-02 2.67967190917923619997154105476511e-02 2.68209752148339686944611415242434e-03
            2.67967190917923619997154105476511e-02 8.78289085336571573314756733452668e-01 2.68209752148339686944611415242434e-03
            8.78289085336571573314756733452668e-01 9.49141955716360924411034716285940e-02 2.68209752148339686944611415242434e-03
            1.23228014736413052365726628067932e-01 8.35184452859199955021551886602538e-01 3.92048297259448702556383636874671e-03
            8.35184452859199955021551886602538e-01 4.15875324043869509793580618861597e-02 3.92048297259448702556383636874671e-03
            4.15875324043869509793580618861597e-02 1.23228014736413052365726628067932e-01 3.92048297259448702556383636874671e-03
            1.74126507274859759766272304659651e-01 4.41306288863040421799333046237734e-02 4.34970113310666813694060905959304e-03
            4.41306288863040421799333046237734e-02 7.81742863838836177237112678994890e-01 4.34970113310666813694060905959304e-03
            7.81742863838836177237112678994890e-01 1.74126507274859759766272304659651e-01 4.34970113310666813694060905959304e-03
            2.72180746005686646782351090223528e-01 3.74687219394429782415301133369212e-02 4.42010181668393668796834816703267e-03
            3.74687219394429782415301133369212e-02 6.90350532054870402731694412068464e-01 4.42010181668393668796834816703267e-03
            6.90350532054870402731694412068464e-01 2.72180746005686646782351090223528e-01 4.42010181668393668796834816703267e-03
            9.90063515075153971478627568103548e-02 8.19687967813020335050566700374475e-02 4.81456610994497949396198066551733e-03
            8.19687967813020335050566700374475e-02 8.19024851711182666491595227853395e-01 4.81456610994497949396198066551733e-03
            8.19024851711182666491595227853395e-01 9.90063515075153971478627568103548e-02 4.81456610994497949396198066551733e-03
            2.18932787550637497631100814032834e-01 7.39314412421514188800131250900449e-01 4.90162630494199013514444729366915e-03
            7.39314412421514188800131250900449e-01 4.17528000278483135687679350667167e-02 4.90162630494199013514444729366915e-03
            4.17528000278483135687679350667167e-02 2.18932787550637497631100814032834e-01 4.90162630494199013514444729366915e-03
            3.85457856195294268353279676375678e-01 3.70114098745062680695383505735663e-02 4.97140868791443122504114526805097e-03
            3.70114098745062680695383505735663e-02 5.77530733930199491332757588679669e-01 4.97140868791443122504114526805097e-03
            5.77530733930199491332757588679669e-01 3.85457856195294268353279676375678e-01 4.97140868791443122504114526805097e-03
            4.53931532973698426136621719706454e-01 5.07124120567042480089980927004945e-01 5.37412036835000720552679709385302e-03
            5.07124120567042480089980927004945e-01 3.89443464592591492845485845464282e-02 5.37412036835000720552679709385302e-03
            3.89443464592591492845485845464282e-02 4.53931532973698426136621719706454e-01 5.37412036835000720552679709385302e-03
            3.31414067992492400449577871768270e-01 6.27872100319265880052910233644070e-01 5.38763633180223099061745273274937e-03
            6.27872100319265880052910233644070e-01 4.07138316882417194975118945876602e-02 5.38763633180223099061745273274937e-03
            4.07138316882417194975118945876602e-02 3.31414067992492400449577871768270e-01 5.38763633180223099061745273274937e-03
            1.75418478700220814570087668471388e-01 1.09788999485381039145082127106434e-01 5.89154887641259519814873613086093e-03
            1.09788999485381039145082127106434e-01 7.14792521814398118529254588793265e-01 5.89154887641259519814873613086093e-03
            7.14792521814398118529254588793265e-01 1.75418478700220814570087668471388e-01 5.89154887641259519814873613086093e-03
            2.99424606258227421484718888677889e-01 8.95177416056505692587563771667192e-02 6.07838382705238374631440123607717e-03
            8.95177416056505692587563771667192e-02 6.11057652136121953745373502897564e-01 6.07838382705238374631440123607717e-03
            6.11057652136121953745373502897564e-01 2.99424606258227421484718888677889e-01 6.07838382705238374631440123607717e-03
            1.64438284214270008520841770405241e-01 7.35720413124063998822066423599608e-01 6.47285261806641171178355875781563e-03
            7.35720413124063998822066423599608e-01 9.98413026616660204126674216240644e-02 6.47285261806641171178355875781563e-03
            9.98413026616660204126674216240644e-02 1.64438284214270008520841770405241e-01 6.47285261806641171178355875781563e-03
            3.69724758939286801684431793546537e-01 4.63127807453672823978507722131326e-01 6.77744987373955429243999049049307e-03
            4.63127807453672823978507722131326e-01 1.67147433607040374337060484322137e-01 6.77744987373955429243999049049307e-03
            1.67147433607040374337060484322137e-01 3.69724758939286801684431793546537e-01 6.77744987373955429243999049049307e-03
            4.06369061777283435699104074956267e-01 9.07167861113062495848069488602050e-02 7.33791624653477511652788223273092e-03
            9.07167861113062495848069488602050e-02 5.02914152111410217571574321482331e-01 7.33791624653477511652788223273092e-03
            5.02914152111410217571574321482331e-01 4.06369061777283435699104074956267e-01 7.33791624653477511652788223273092e-03
            2.67420664299267585306552064139396e-01 6.32902725169850244313352050085086e-01 7.71022722919994900336870813362111e-03
            6.32902725169850244313352050085086e-01 9.96766105308821703800958857755177e-02 7.71022722919994900336870813362111e-03
            9.96766105308821703800958857755177e-02 2.67420664299267585306552064139396e-01 7.71022722919994900336870813362111e-03
            3.83204107909751179317225933118607e-01 5.21623720050690464766773857263615e-01 7.75799889707759742635273525479533e-03
            5.21623720050690464766773857263615e-01 9.51721720395583004048489783599507e-02 7.75799889707759742635273525479533e-03
            9.51721720395583004048489783599507e-02 3.83204107909751179317225933118607e-01 7.75799889707759742635273525479533e-03
            1.78724492320244954246533097830252e-01 1.92031135757774140682130337154376e-01 7.83129465300495472557251730449934e-03
            1.92031135757774140682130337154376e-01 6.29244371921980905071336565015372e-01 7.83129465300495472557251730449934e-03
            6.29244371921980905071336565015372e-01 1.78724492320244954246533097830252e-01 7.83129465300495472557251730449934e-03
            2.82829128593355183518553985777544e-01 5.37642437490313485071169452567119e-01 7.96376875542390005868753632967127e-03
            5.37642437490313485071169452567119e-01 1.79528433916331331410276561655337e-01 7.96376875542390005868753632967127e-03
            1.79528433916331331410276561655337e-01 2.82829128593355183518553985777544e-01 7.96376875542390005868753632967127e-03
            2.70606479254229259634456639105338e-01 1.61339950052654446110977914941031e-01 8.05176651503982168667228336289554e-03
            1.61339950052654446110977914941031e-01 5.68053570693116238743414214695804e-01 8.05176651503982168667228336289554e-03
            5.68053570693116238743414214695804e-01 2.70606479254229259634456639105338e-01 8.05176651503982168667228336289554e-03
            3.70874346744923255592141231318237e-01 1.64548573538222442369516329563339e-01 8.32904021286562899761829470435259e-03
            1.64548573538222442369516329563339e-01 4.64577079716854302038342439118423e-01 8.32904021286562899761829470435259e-03
            4.64577079716854302038342439118423e-01 3.70874346744923255592141231318237e-01 8.32904021286562899761829470435259e-03
            3.63840811889283188840238381089875e-01 3.83367191770047632815021643182263e-01 9.67653974618146520136896526764758e-03
            3.83367191770047632815021643182263e-01 2.52791996340669178344739975727862e-01 9.67653974618146520136896526764758e-03
            2.52791996340669178344739975727862e-01 3.63840811889283188840238381089875e-01 9.67653974618146520136896526764758e-03
            2.75735321862771332046548877769965e-01 2.52760449512089535595293909864267e-01 1.02296793188677258290564253684352e-02
            2.52760449512089535595293909864267e-01 4.71504228625139076847005981107941e-01 1.02296793188677258290564253684352e-02
            4.71504228625139076847005981107941e-01 2.75735321862771332046548877769965e-01 1.02296793188677258290564253684352e-02
            ];
        
    case  25
        
        % ALG. DEG.:   25
        % PTS CARD.:  117
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.4e-16
        
        xyw=[
            1.31881771474051147363670821732740e-02 3.07956384011914569703849409165741e-03 2.63473905485520402090793279370473e-04
            3.07956384011914569703849409165741e-03 9.83732259012475740433956161723472e-01 2.63473905485520402090793279370473e-04
            9.83732259012475740433956161723472e-01 1.31881771474051147363670821732740e-02 2.63473905485520402090793279370473e-04
            2.72596213613051892646343965509459e-03 1.71043047595741092015142470472711e-02 2.69189285624497009162536809157018e-04
            1.71043047595741092015142470472711e-02 9.80169733104295337611233662755694e-01 2.69189285624497009162536809157018e-04
            9.80169733104295337611233662755694e-01 2.72596213613051892646343965509459e-03 2.69189285624497009162536809157018e-04
            6.29840020924785359834885412055883e-02 9.31371984168810507220825911645079e-01 8.56278646941008823867402810492422e-04
            9.31371984168810507220825911645079e-01 5.64401373871092904010993152041920e-03 8.56278646941008823867402810492422e-04
            5.64401373871092904010993152041920e-03 6.29840020924785359834885412055883e-02 8.56278646941008823867402810492422e-04
            5.20238006731535140847455522816745e-02 7.43791496086179931140902610309240e-03 9.56166433485291979849085564069355e-04
            7.43791496086179931140902610309240e-03 9.40538284365984744717081866838271e-01 9.56166433485291979849085564069355e-04
            9.40538284365984744717081866838271e-01 5.20238006731535140847455522816745e-02 9.56166433485291979849085564069355e-04
            2.20200206141671100956713758023398e-01 7.76414334424490371056037929520244e-01 9.94872257003353438475912184912886e-04
            7.76414334424490371056037929520244e-01 3.38545943383850023167269682744518e-03 9.94872257003353438475912184912886e-04
            3.38545943383850023167269682744518e-03 2.20200206141671100956713758023398e-01 9.94872257003353438475912184912886e-04
            3.04395561568152825393518412511185e-02 9.44856451527789409361446359980619e-01 1.35150087580622967585453064742751e-03
            9.44856451527789409361446359980619e-01 2.47039923153953289158835104899481e-02 1.35150087580622967585453064742751e-03
            2.47039923153953289158835104899481e-02 3.04395561568152825393518412511185e-02 1.35150087580622967585453064742751e-03
            1.13993174377526096385793152876431e-01 7.47102931577944221613885034116720e-03 1.41188107103241760817424754748117e-03
            7.47102931577944221613885034116720e-03 8.78535796306694427570960215234663e-01 1.41188107103241760817424754748117e-03
            8.78535796306694427570960215234663e-01 1.13993174377526096385793152876431e-01 1.41188107103241760817424754748117e-03
            3.42295210188022336339486173528712e-01 6.52613563397589468983994720474584e-01 1.45757458623657653630500963259919e-03
            6.52613563397589468983994720474584e-01 5.09122641438819467651910599670373e-03 1.45757458623657653630500963259919e-03
            5.09122641438819467651910599670373e-03 3.42295210188022336339486173528712e-01 1.45757458623657653630500963259919e-03
            1.30473991950559503605333588893700e-01 8.61524966076299802963944785005879e-01 1.57311439115006461778034640985879e-03
            8.61524966076299802963944785005879e-01 8.00104197314066567514601047150791e-03 1.57311439115006461778034640985879e-03
            8.00104197314066567514601047150791e-03 1.30473991950559503605333588893700e-01 1.57311439115006461778034640985879e-03
            2.00179308343867129460491582904069e-01 6.50781868302953165478230701523898e-03 1.62719887443376113288218132169050e-03
            6.50781868302953165478230701523898e-03 7.93312872973103377916004319558851e-01 1.62719887443376113288218132169050e-03
            7.93312872973103377916004319558851e-01 2.00179308343867129460491582904069e-01 1.62719887443376113288218132169050e-03
            3.07372363215433874739090924776974e-01 6.35240709459219433929977327579763e-03 1.86153289074892486741619190837582e-03
            6.35240709459219433929977327579763e-03 6.86275229689973897961863258387893e-01 1.86153289074892486741619190837582e-03
            6.86275229689973897961863258387893e-01 3.07372363215433874739090924776974e-01 1.86153289074892486741619190837582e-03
            2.76217326622710424022244524167036e-01 7.06370557783410402130641614348860e-01 2.15315736964076477263918540927534e-03
            7.06370557783410402130641614348860e-01 1.74121155938792293582650927419309e-02 2.15315736964076477263918540927534e-03
            1.74121155938792293582650927419309e-02 2.76217326622710424022244524167036e-01 2.15315736964076477263918540927534e-03
            4.22935483077426899178874464269029e-01 7.56275469533758924745026064329068e-03 2.26256707948115651576892481955383e-03
            7.56275469533758924745026064329068e-03 5.69501762227235475144482279574731e-01 2.26256707948115651576892481955383e-03
            5.69501762227235475144482279574731e-01 4.22935483077426899178874464269029e-01 2.26256707948115651576892481955383e-03
            4.53227003366671432615930825704709e-01 5.38998045885384025055486745259259e-01 2.28124961065890403608125147627561e-03
            5.38998045885384025055486745259259e-01 7.77495074794454232858242903603241e-03 2.28124961065890403608125147627561e-03
            7.77495074794454232858242903603241e-03 4.53227003366671432615930825704709e-01 2.28124961065890403608125147627561e-03
            1.90195202693624082490586602034455e-01 7.84036511910398581370884585339809e-01 2.34013831030492700652212434420107e-03
            7.84036511910398581370884585339809e-01 2.57682853959773083829531969968230e-02 2.34013831030492700652212434420107e-03
            2.57682853959773083829531969968230e-02 1.90195202693624082490586602034455e-01 2.34013831030492700652212434420107e-03
            3.61718299806125034856485456202790e-02 8.44564005292429142990329182794085e-02 2.82027799077192147664261412387532e-03
            8.44564005292429142990329182794085e-02 8.79371769490144616909788055636454e-01 2.82027799077192147664261412387532e-03
            8.79371769490144616909788055636454e-01 3.61718299806125034856485456202790e-02 2.82027799077192147664261412387532e-03
            7.60794388659036535660362687849556e-02 4.05076326129555214228261661446595e-02 2.88921152538740375753656408619463e-03
            4.05076326129555214228261661446595e-02 8.83412928521140838888925372884842e-01 2.88921152538740375753656408619463e-03
            8.83412928521140838888925372884842e-01 7.60794388659036535660362687849556e-02 2.88921152538740375753656408619463e-03
            1.49945977487857845522611910382693e-01 3.58350589330340557614462682067824e-02 3.28957067535762346516259313489172e-03
            3.58350589330340557614462682067824e-02 8.14218963579108057082578397967154e-01 3.28957067535762346516259313489172e-03
            8.14218963579108057082578397967154e-01 1.49945977487857845522611910382693e-01 3.28957067535762346516259313489172e-03
            1.50955341179002944240750139215379e-01 7.96913702682935043419831799837993e-01 3.36482658292138088473866019967318e-03
            7.96913702682935043419831799837993e-01 5.21309561380619568282668296888005e-02 3.36482658292138088473866019967318e-03
            5.21309561380619568282668296888005e-02 1.50955341179002944240750139215379e-01 3.36482658292138088473866019967318e-03
            2.44783310976574935979854785728094e-01 3.26897799308286454711414137364045e-02 3.78777121793418512785134311116053e-03
            3.26897799308286454711414137364045e-02 7.22526909092596425487897704442730e-01 3.78777121793418512785134311116053e-03
            7.22526909092596425487897704442730e-01 2.44783310976574935979854785728094e-01 3.78777121793418512785134311116053e-03
            1.70674433667943242776132706239878e-01 7.87558693381886776307965192245319e-02 4.07291752316681424095357044734556e-03
            7.87558693381886776307965192245319e-02 7.50569696993868107348646390164504e-01 4.07291752316681424095357044734556e-03
            7.50569696993868107348646390164504e-01 1.70674433667943242776132706239878e-01 4.07291752316681424095357044734556e-03
            3.71942611691943147977923445068882e-01 5.93428569491770030452926221187226e-01 4.54091761981328164571314331965368e-03
            5.93428569491770030452926221187226e-01 3.46288188162868770803015650017187e-02 4.54091761981328164571314331965368e-03
            3.46288188162868770803015650017187e-02 3.71942611691943147977923445068882e-01 4.54091761981328164571314331965368e-03
            3.51792010432223745119983959739329e-01 3.63528731441380772992921777131414e-02 4.70044765215418044318473889120469e-03
            3.63528731441380772992921777131414e-02 6.11855116423638212275193382083671e-01 4.70044765215418044318473889120469e-03
            6.11855116423638212275193382083671e-01 3.51792010432223745119983959739329e-01 4.70044765215418044318473889120469e-03
            9.84400611472149716929536111820198e-02 9.69591386055578818847777711198432e-02 4.76200227203537641840203065157766e-03
            9.69591386055578818847777711198432e-02 8.04600800247227132544480809883680e-01 4.76200227203537641840203065157766e-03
            8.04600800247227132544480809883680e-01 9.84400611472149716929536111820198e-02 4.76200227203537641840203065157766e-03
            2.60865026298107871571829718959634e-01 6.84089653184228985161041691753780e-01 5.05360407263360893181936006612887e-03
            6.84089653184228985161041691753780e-01 5.50453205176631987782798205444124e-02 5.05360407263360893181936006612887e-03
            5.50453205176631987782798205444124e-02 2.60865026298107871571829718959634e-01 5.05360407263360893181936006612887e-03
            3.66582144327067316336155045064515e-01 5.50781902551756963326567984040594e-01 5.42956340385742069337693394004418e-03
            5.50781902551756963326567984040594e-01 8.26359531211756648261257396370638e-02 5.42956340385742069337693394004418e-03
            8.26359531211756648261257396370638e-02 3.66582144327067316336155045064515e-01 5.42956340385742069337693394004418e-03
            4.68735987743839010999380434441264e-01 4.11368221710290485337857546710438e-02 5.58449209314384412833343063198299e-03
            4.11368221710290485337857546710438e-02 4.90127190085131891894576483537094e-01 5.58449209314384412833343063198299e-03
            4.90127190085131891894576483537094e-01 4.68735987743839010999380434441264e-01 5.58449209314384412833343063198299e-03
            2.57617931613541240043474545018398e-01 8.24865172853232742067319804846193e-02 5.88614843261607567637128113346989e-03
            8.24865172853232742067319804846193e-02 6.59895551101135513505369090125896e-01 5.88614843261607567637128113346989e-03
            6.59895551101135513505369090125896e-01 2.57617931613541240043474545018398e-01 5.88614843261607567637128113346989e-03
            1.84517952412833835529326620417123e-01 7.12575925803732079799601706326939e-01 6.07120200423857687282991335564475e-03
            7.12575925803732079799601706326939e-01 1.02906121783434056915496057627024e-01 6.07120200423857687282991335564475e-03
            1.02906121783434056915496057627024e-01 1.84517952412833835529326620417123e-01 6.07120200423857687282991335564475e-03
            2.81209118651227718554963530550594e-01 6.05257308197521104453642237785971e-01 6.27634946334984338001117265548601e-03
            6.05257308197521104453642237785971e-01 1.13533573151251121480243000405608e-01 6.27634946334984338001117265548601e-03
            1.13533573151251121480243000405608e-01 2.81209118651227718554963530550594e-01 6.27634946334984338001117265548601e-03
            1.73591491675633341484896732254128e-01 1.42209318699190873092064180127636e-01 6.39072063804919411611482615853674e-03
            1.42209318699190873092064180127636e-01 6.84199189625175785423039087618235e-01 6.39072063804919411611482615853674e-03
            6.84199189625175785423039087618235e-01 1.73591491675633341484896732254128e-01 6.39072063804919411611482615853674e-03
            4.42362027995821582848634534457233e-01 4.54988327657975588635963504202664e-01 6.62342069381232582264695807339194e-03
            4.54988327657975588635963504202664e-01 1.02649644346202828515401961340103e-01 6.62342069381232582264695807339194e-03
            1.02649644346202828515401961340103e-01 4.42362027995821582848634534457233e-01 6.62342069381232582264695807339194e-03
            3.59368226555797587451479557785206e-01 9.07953156997939186956259050020890e-02 6.91412078175607583419015611525538e-03
            9.07953156997939186956259050020890e-02 5.49836457744408479975106729398249e-01 6.91412078175607583419015611525538e-03
            5.49836457744408479975106729398249e-01 3.59368226555797587451479557785206e-01 6.91412078175607583419015611525538e-03
            1.82134268926742365346527208203042e-01 2.28667042140729831478651590259688e-01 7.96410608107515731457493046718810e-03
            2.28667042140729831478651590259688e-01 5.89198688932527803174821201537270e-01 7.96410608107515731457493046718810e-03
            5.89198688932527803174821201537270e-01 1.82134268926742365346527208203042e-01 7.96410608107515731457493046718810e-03
            2.67343042736282066051956007868284e-01 1.56184413818651390837288772672764e-01 8.37297856925192478250163929942573e-03
            1.56184413818651390837288772672764e-01 5.76472543445066598621906450716779e-01 8.37297856925192478250163929942573e-03
            5.76472543445066598621906450716779e-01 2.67343042736282066051956007868284e-01 8.37297856925192478250163929942573e-03
            3.65290320906562715475018876531976e-01 1.73792802318537387140651162553695e-01 9.38730527689661531365317159725237e-03
            1.73792802318537387140651162553695e-01 4.60916876774899841873178729656502e-01 9.38730527689661531365317159725237e-03
            4.60916876774899841873178729656502e-01 3.65290320906562715475018876531976e-01 9.38730527689661531365317159725237e-03
            3.45064826601384877502454173736623e-01 4.82299125389173410916043849283597e-01 9.86900254009348727735062567489877e-03
            4.82299125389173410916043849283597e-01 1.72636048009441656070350745721953e-01 9.86900254009348727735062567489877e-03
            1.72636048009441656070350745721953e-01 3.45064826601384877502454173736623e-01 9.86900254009348727735062567489877e-03
            2.65562227862875310790968796936795e-01 2.56802629703844209618779359516338e-01 1.01431421793431401218166243438645e-02
            2.56802629703844209618779359516338e-01 4.77635142433280479590251843546866e-01 1.01431421793431401218166243438645e-02
            4.77635142433280479590251843546866e-01 2.65562227862875310790968796936795e-01 1.01431421793431401218166243438645e-02
            3.62956385501777445412585620942991e-01 3.71550947373690598762863146475866e-01 1.08126717889738110239816748503472e-02
            3.71550947373690598762863146475866e-01 2.65492667124531955824551232581143e-01 1.08126717889738110239816748503472e-02
            2.65492667124531955824551232581143e-01 3.62956385501777445412585620942991e-01 1.08126717889738110239816748503472e-02
            ];
        
        
    otherwise
        xyw=[];
end




pointset_stats=[
    10   24 2.08e-15    0    0
    11   27 9.92e-16    0    0
    12   31 2.21e-15    0    3
    13   36 3.05e-16    0    0
    14   40 9.51e-16    0    3
    15   46 5.00e-16    0    0
    16   51 1.78e-15    0    6
    17   57 5.83e-16    0    0
    18   64 4.86e-16    1    0
    19   70 4.16e-16    0    0
    20   78 4.89e-16    0    0
    21   85 4.64e-16    0    0
    22   93 7.56e-16    0    0
    23  100 6.66e-16    0    0
    24  109 7.81e-16    0    0
    25  117 5.41e-16    0    0
    ];




