
function [xyw_bar,pointset_stats]=set_papanicolopulos_B_barycentric(deg)

%--------------------------------------------------------------------
% CUBATURE RULE ON THE SIMPLEX [0 0; 0 1; 1 0] WITH ADE EQUAL TO deg.
% THE CUBATURE RULE IS DESCRIBED IN BARYCENTRICAL FORM.
%--------------------------------------------------------------------
% INPUT:
% deg: algebraic degree of precision (ADE) of the rule.
%--------------------------------------------------------------------
% OUTPUT:
% xyw_bar: MATRIX OF NODES xyw(:,1:3) IN BARYCENTRICAL COOR. AND WEIGHTS xyw(:,4).
%      REMARK: The sum of the weights makes 1.
%
% pointset_stats:
%    statistics of the pointset stored in a matrix. Each rows contains
%    degree, cardinality of the full pointset, moment error w.r.t.
%    orthn. Dubiner basis, number of points with negative weights,
%    number of points outside the domain.
%--------------------------------------------------------------------
% SOURCE:
% S-A. Papanicolopulos
% "Efficient computation of cubature rules with application to new
%  asymmetric rules on the triangle"
% JCAM 204 (2016), pp. 77-83.
%--------------------------------------------------------------------
% | DEG | CARD | W- | PT.O|   M.E.   |Q.|
%--------------------------------------------------------------------
% |  11 |  26 |   0 |   1 | 4.72e-16 |P0|
% |  20 |  77 |   0 |   8 | 1.45e-15 |P0|
% |  22 |  92 |   0 |  11 | 1.72e-15 |P0|
%--------------------------------------------------------------------
% LEGEND:
%--------------------------------------------------------------------
% DEG  : algebraic degree of precision
% CARD : cardinality of the pointset
% W-   : negative weights
% PT.O : points outside the simplex [0 0; 1 0; 0 1].
% M.E. : maximum moment error in inf norm.
% Q.: quality of the rule.
%        PI: pos. weights, points in simplex
%        NI: some neg. weights, points in simplex
%        P0: pos. weights, some points out of the simplex
%        N0: some neg. weights, some points out of the simplex
%--------------------------------------------------------------------

switch deg
    
    case  11
        
        % ALG. DEG.:   11
        % PTS CARD.:   26
        % NEG. W.  :    0
        % OUT PTS. :    1
        % M.E.INF. : 4.7e-16
        
        xyw_bar=[
            6.25873806592363246537047416495625e-01 -4.44983835604058006185645979257970e-02 4.18624576968042561020411085337400e-01 1.77034293515736826200857034763203e-03
            2.90632953572616987980570257832369e-02 9.40848476278976941777898446161998e-01 3.00882283637613490157036721939221e-02 1.14071815427992453995109656261775e-02
            9.33058339148674420471252233255655e-01 3.34630956930927966896938130503258e-02 3.34785651582327828390539536940196e-02 1.44501488320227886724689270181443e-02
            3.18590326985022029293581624642684e-02 3.55146752201139018212394660167774e-02 9.32626292081383923004977987147868e-01 1.45717137218703551848619781594607e-02
            1.62482633218668293118014389619930e-01 1.39846877219226006783259208532399e-02 8.23532679059409078448084073897917e-01 1.50827170503701740028601463450286e-02
            1.32184617612812003312194875093155e-02 6.42788826815432368100289295398397e-01 3.43992711423286467997684212605236e-01 1.73815046654114679913760710405768e-02
            5.91829479333409902430673810158623e-01 3.93141694274107222906167180553894e-01 1.50288263924828746631590092874831e-02 1.97880489137955695166937175599742e-02
            8.05575896789687551091674322378822e-01 2.46651880397084008467167848266399e-02 1.69758915170604041122714988887310e-01 2.30346283177385001417203369555864e-02
            8.04321495466082558856157902482664e-01 1.70121715342375495749749347851321e-01 2.55567891915419453940927496660152e-02 2.40742956118628545492121872939606e-02
            1.51797774113621691816078396186640e-01 8.18070855798455953689085617952514e-01 3.01313700879223267392603702319320e-02 2.57822931384248804631553753097251e-02
            2.90692872577444014203784661276586e-02 1.77929391940772008151583349899738e-01 7.93001320801483666755871126952115e-01 2.78545365971359509815741262173105e-02
            3.42125596440217977223241518913710e-02 8.14117937083535969655656572285807e-01 1.51669503272442218744231468008365e-01 2.80274085579395662581880799280043e-02
            2.48467017693192009086811111728821e-02 3.97321198592043678399221562358434e-01 5.77832099638637020078135719813872e-01 3.01337505350859312380773502582088e-02
            3.53270314298278487186166785249952e-01 6.15916760551632136433397590735694e-01 3.08129251500893763804356240143534e-02 3.53256132449768986414362359482766e-02
            3.61781261878188875602546659138170e-01 3.48462422413524028352860284485359e-02 6.03372495880458825645575871021720e-01 3.96262736802082612963538110761874e-02
            7.25512890517397424083867463195929e-01 1.34200454674975294677707893242768e-01 1.40286654807627281238424643561302e-01 4.70490776723007098070183928939514e-02
            1.52687758377548998822703651967458e-01 1.08274570877667397628130174780381e-01 7.39037670744783659060317404509988e-01 4.76266345794480461250941516482271e-02
            5.76760918709025127704137503314996e-01 3.29909545553079719404365732771112e-01 9.33295357378951528914967639138922e-02 4.91237863062456148188772431240068e-02
            9.57812464558655957125310465016810e-02 5.53708279551195170498090192268137e-01 3.50510473992939219911590953415725e-01 5.58925170867282730680081215268729e-02
            1.66286918287333995714050161041087e-01 6.79990236477728493547090238280362e-01 1.53722845234937510738859600678552e-01 5.69262728053029770647341933909047e-02
            5.80931106721034651307888907467714e-01 5.93349142682009986216051800056448e-02 3.59733979010764370887187624248327e-01 5.73269518700233599028948106024473e-02
            1.33154613412877709244241941632936e-01 2.98286541434518104320972042842186e-01 5.68558845152604241945937246782705e-01 6.34711759189008906112405838939594e-02
            3.61350975086914882883348809627932e-01 4.88966240018430697844564747356344e-01 1.49682784894654474783237674273551e-01 6.38061809796458406784935846189910e-02
            3.26342158377320279427635796309914e-01 1.74983427068089586464694207279535e-01 4.98674414554590161863245612039464e-01 7.42559452939303143992333389178384e-02
            4.93348382710591615296635836784844e-01 2.37360633371601903451875159589690e-01 2.69290983917806536762640234883293e-01 7.75897797681213696741053809091682e-02
            2.60838798471289823321939138622838e-01 4.01888923157516897433083613577764e-01 3.37272278371193223733826016541570e-01 7.86212203745528109832818586255598e-02
            ];
        
    case  20
        
        % ALG. DEG.:   20
        % PTS CARD.:   77
        % NEG. W.  :    0
        % OUT PTS. :    8
        % M.E.INF. : 1.5e-15
        
        xyw_bar=[
            -2.13972282618556192135272908672050e-01 8.85270494328899482106010054849321e-01 3.28701788289656682273687238193816e-01 3.32720318999303078765234592225394e-09
            1.14219528684296989240465336479247e+00 -1.72274034387398494727605680054694e-01 3.00787475444286023229523152622278e-02 1.15227306672511206906615822617712e-08
            5.99214262445217360486537927499739e-01 -1.26598687399647497064947287981340e-01 5.27384424954430164333984976110514e-01 6.11934497644042595088957809702779e-08
            1.24379871889390994366131337756087e-01 -1.18785618275940599297690880575828e-01 9.94405746386549549420408311561914e-01 6.71689836409761573790113482924913e-08
            6.53183349102199051205985824708478e-02 1.02882448539515780616682150139241e+00 -9.41428203053776835318444682343397e-02 8.55151997588337884174133993880895e-07
            -5.36667090855443978947292293923965e-02 1.64022325296424409390994014756870e-01 8.89644383789120141159401100594550e-01 1.51752319457325498126528187659012e-05
            2.85608320415044192319697913262644e-01 -3.44098409098497967639751493607037e-02 7.48801520494805639138746755634202e-01 7.65101570495463931091684184337964e-05
            9.36080843693492092860708453372354e-01 8.34052574449410943913107985281385e-02 -1.94861011384331872520192519004922e-02 2.17651898683011762927058674321756e-04
            8.03052528439718082964304812776390e-01 1.94561633923371501975196906641941e-01 2.38583763691041506049828058166895e-03 1.43156144316582546564819988788031e-03
            1.20188880734230004593277030267018e-02 1.22773002304658995081920735970016e-02 9.75703811696111089624139367515454e-01 1.93144265473825949702701976207209e-03
            1.17885985793459002052285811146248e-02 9.74683306669172688785351965634618e-01 1.35280947514814231524837850884069e-02 2.08348597472964329122002702376903e-03
            9.72419471438029980880912717111642e-01 1.51595907332457008331871861628315e-02 1.24209378287243182859000967255270e-02 2.45043394363832168700301217256765e-03
            2.59582889842235209343357382749673e-01 7.37406178647960541816530621872516e-01 3.01093150980424884011199537781067e-03 3.17523433288238992461272047762577e-03
            1.13901722279699992718127177226961e-02 6.38319730537900004163631706433080e-02 9.24777854718240033271570155193331e-01 4.13324939705091848329843173814879e-03
            6.24536748324725984882199725234386e-02 1.19686440555381006650703668015012e-02 9.25577681111989347684243512048852e-01 4.23938988379203400558292713640185e-03
            5.36487886917886624971174569509458e-01 4.58640643576341988651989822756150e-01 4.87146950577138637683560773439240e-03 4.24624648435661163764498482464660e-03
            1.15287726880495995651632057388269e-02 9.18451304554263936452684902178589e-01 7.00199227576864258182354205928277e-02 4.57849937718211169590531994799676e-03
            9.21229627801570027934019435633672e-01 1.37267427812455006935765666753468e-02 6.50436294171844731071274736677879e-02 5.06809826713209860687969765535854e-03
            6.09542503069403981963425565027137e-02 9.22060171859679145178745329758385e-01 1.69855778333805051971694410894997e-02 5.73541752590444719323015121403841e-03
            1.49807778345135700348578211560380e-01 1.17795744693112991452466076225392e-02 8.38412647185552972750599565188168e-01 6.11502851381981460399117622728227e-03
            7.07889807463940590182005507813301e-01 8.34091008803320051179230887328231e-03 2.83769282448026183285350043661310e-01 6.14046619790678885331125513857842e-03
            8.32342815256551848968058493483113e-01 1.09109589105933994096497841042037e-02 1.56746225832854746418121294482262e-01 6.15455433207981461585234228550689e-03
            1.19250456031763998887562649997562e-02 8.21808561973290996327534685406135e-01 1.66266392423532582967027337872423e-01 6.89487152224487788443330060772496e-03
            1.45547229195401611745452896684583e-01 8.40701488801081198154463436367223e-01 1.37512820035171623445080513192806e-02 7.02474576081652674064415009524964e-03
            9.53344909862949979606483452698740e-03 4.10486940081249973566457356355386e-01 5.79979610820120505820796097395942e-01 7.17465503188439296389233845729905e-03
            6.79668341434463973982360585068818e-01 3.10086648021363597838018222319079e-01 1.02450105441724281796211926121032e-02 7.25071792347643968856818474932879e-03
            9.89232730221640058076992829683149e-03 5.56673185362967171130321730743162e-01 4.33434487334816376247204061655793e-01 7.28528984220321636916528618144184e-03
            5.53456071686292960798425610846607e-01 1.04547658690504995104841867714640e-02 4.36089162444656563977218866057228e-01 8.04728227525123435137643923553696e-03
            8.09484224243874894710870648850687e-01 1.70682203625222100873415342903172e-01 1.98335721309030044157140082461410e-02 8.29517431446462685795673053235078e-03
            9.02013507663122937607624862721423e-01 7.46138877131691941047009208887175e-02 2.33726046237078682876742163898598e-02 8.36349591977037301604447350200644e-03
            1.29694574942774993459027044195864e-02 2.70557639318955600593596955150133e-01 7.16472903186766929550799432036001e-01 8.63288764034306904671112192772853e-03
            5.86275629455060981465308600490971e-02 6.12222576866121026561984308500541e-02 8.80150179367881868586209748173133e-01 8.63386603107164224413239139721554e-03
            3.92424909662620502448504566928023e-01 5.95206388831197807931516763346735e-01 1.23687015061817451311299009830691e-02 8.81867751376813960306755291185254e-03
            1.37405978198458008504578842234878e-02 6.94659120250108452765402944351081e-01 2.91600281930045723832733983726939e-01 9.44688549540443310570392299041487e-03
            1.87516304312781989516167868714547e-02 1.51227587577556388254151897854172e-01 8.30020781991165357283080084016547e-01 9.56916740283403170763687484168258e-03
            4.02956369953581583498447571400902e-01 1.37359323322951993456841890406395e-02 5.83307697714123163379440484277438e-01 1.01477249065513083076117695213725e-02
            1.37650638016870607849284624535358e-01 5.64581638297467006126773014784703e-02 8.05891198153382726232507593522314e-01 1.06018367078154502475273446293613e-02
            5.98726536103752968176472393224685e-02 8.58559558597151051273499433591496e-01 8.15677877924736449699594231788069e-02 1.11992726408217946332257497488172e-02
            6.24882352893667780691089319589082e-01 4.16470200265496998115466453782574e-02 3.33470627079782533375151842847117e-01 1.20086917263369265251871098598713e-02
            2.62938174339716002059219590591965e-01 1.99626935996523001415337006392292e-02 7.17099132060631694329799756815191e-01 1.20653065639470969544477796375759e-02
            8.38049083719112686274854695511749e-01 6.47623246114735068301726528261497e-02 9.71885916694138068949726516621013e-02 1.31214199223905307600812619739372e-02
            2.57533995834666518121025546861347e-01 7.06047344861891468248416003916645e-01 3.64186593034420136305584492220078e-02 1.34650124083951303749495664874303e-02
            5.35401647857205564839944145205664e-01 4.29012681315519783797896025134833e-01 3.55856708272746513621598296595039e-02 1.37859192899982673646830377833794e-02
            7.43231502264049992412253686779877e-01 5.13341047302287994957836758658232e-02 2.05434393005721194214174829539843e-01 1.41321111350623893759825833171817e-02
            4.88883387859557969057355819586519e-02 5.21869746031161985122537316783564e-01 4.29241915182882238788408812979469e-01 1.44051328146765546950325287411943e-02
            6.24726931520614980897221357736271e-02 7.57815811962161856030206763534807e-01 1.79711494885776645880071100691566e-01 1.52644419061392436209878553654562e-02
            1.43597231050559187570669905653631e-01 7.85973813626918538233212530030869e-01 7.04289553225222464405419486865867e-02 1.53012865423340499448912055413530e-02
            6.69101112127524655370791606401326e-01 2.76326273581061820916460192165687e-01 5.45726142914135237127482014329871e-02 1.64745514173273641111983067730762e-02
            7.65185966369098746753252271446399e-01 1.55100092624901292337469271842565e-01 7.97139410059999609092784567110357e-02 1.65322644947739592902191674284040e-02
            8.85954672172785950845508295969921e-02 1.37178277163375605951500801893417e-01 7.74226255619345771208372752880678e-01 1.66249745922861374980428195158311e-02
            5.27740002133011976148679877951508e-02 3.82366757870937379326647942434647e-01 5.64859241915761423058484069770202e-01 1.68291134834775146789098698718590e-02
            6.80360399128474013785705665213754e-02 2.47691298374512897106924924628402e-01 6.84272661712639673758928893221309e-01 1.79328491360184923142373492055412e-02
            6.98062102092747993609123113856185e-02 6.31595957662000828669590646313736e-01 2.98597832128724371969497042300645e-01 1.87804488445696624032699162398785e-02
            2.07987112156463094247982326123747e-01 9.24058022638531012526996732958651e-02 6.99607085579683762865954577137018e-01 1.88201673872226069028634753976803e-02
            3.90182931303222124697072104027029e-01 5.47411613364379934232317737041740e-01 6.24054553323979410706101589312311e-02 1.88795568046839536135994563892382e-02
            4.76076386842712473423944175010547e-01 6.02551121213251023034374043163552e-02 4.63668501035962410394830612858641e-01 1.88890958270543675434627317599734e-02
            3.34886348256971777548329782803194e-01 7.47904975261002996633763473255385e-02 5.90323154216927936666081677685725e-01 2.05499613524602227354609595977308e-02
            5.86423015392367852349764234531904e-01 1.03489104558458305804791166337964e-01 3.10087880049173869601020214759046e-01 2.06645629372090369746395310812659e-02
            1.46590103668047588847400675149402e-01 6.90221726784093880802117837447440e-01 1.63188169547858530350481487403158e-01 2.11965925333831746446566768327102e-02
            6.89198424849254087654060185741400e-01 1.30677796659157713365928543680639e-01 1.80123778491588198980011270577961e-01 2.12426234364560713452352302965664e-02
            2.56431003885450981005789117261884e-01 6.35951264400860249992319950251840e-01 1.07617731713688824513042163744103e-01 2.14176567591459858508340374783074e-02
            5.19807965736053700922525422356557e-01 3.79116974478602319909015250232187e-01 1.01075059785343979168459327411256e-01 2.19222349382033471210107222759689e-02
            1.18988935286532496182765328285313e-01 4.57943028819293218578678761332412e-01 4.23068035894174243605192486938904e-01 2.31205812869516025287452976044733e-02
            6.13670324239234954433186430833302e-01 2.44024343068565796599500572483521e-01 1.42305332692199248967312996683177e-01 2.41573832983769908044902052779435e-02
            1.56249773792297513841376144227979e-01 5.64568937312600049871491592057282e-01 2.79181288895102408531556648085825e-01 2.42846019891320742678519906121437e-02
            2.55871503699490576710928735337802e-01 5.36571755735847366608481934235897e-01 2.07556740564662112191740561684128e-01 2.44764931883724237704491599743051e-02
            1.70734103415520588331588669461780e-01 1.92078073276151894965835253969999e-01 6.37187823308327572213727307826048e-01 2.46405530626073072941739638963554e-02
            3.79086896953479079996895961812697e-01 4.74425078234563923906819127296330e-01 1.46488024811956996096284910890972e-01 2.48697186282620871788928695877985e-02
            1.37580605927453802905802149325609e-01 3.19431457488133119237971868642489e-01 5.42987936584413022345074750774074e-01 2.49534039306388184809026853372416e-02
            2.18911141386702090549931654095417e-01 4.16189965552626772726796389179071e-01 3.64898893060671136723271956725512e-01 2.64949360223682116310950362958465e-02
            4.34412095811269105549712321590050e-01 1.46976743935933906737290044475230e-01 4.18611160252796932201846402676892e-01 2.69906644734016001874188361853157e-02
            2.96550157207215503429864611462108e-01 1.70010182348781513361757333768765e-01 5.33439660444002927697226823511301e-01 2.70358973052389484070801017878694e-02
            3.23070720478415784882031402958091e-01 4.00096706715143080046459544973914e-01 2.76832572806441135071509052067995e-01 2.73298130840888865711324484664146e-02
            5.30596277259844262275123583094683e-01 2.03039795210256696078587879128463e-01 2.66363927529899013890712922147941e-01 2.77165352238719631117191966040991e-02
            2.48460261553608813267501886912214e-01 2.88066651277510521467206672241446e-01 4.63473087168880693020867056475254e-01 2.83257257388832180133864113713571e-02
            4.58735528925298396973886383420904e-01 3.30663543143968396975651558022946e-01 2.10600927930733261561613289813977e-01 2.84959896838699391907745450680522e-02
            3.76985663109379198765225282841129e-01 2.65396696108143426151571020454867e-01 3.57617640782477375083203696704004e-01 3.16217319212400566064147255929129e-02
            ];
        
    case  22
        
        % ALG. DEG.:   22
        % PTS CARD.:   92
        % NEG. W.  :    0
        % OUT PTS. :   11
        % M.E.INF. : 1.7e-15
        
        xyw_bar=[
            4.64982138245242704321924520627363e-01 7.11150575650459360055322122207144e-01 -1.76132713895702064377246642834507e-01 4.53956399013490879146945055082514e-09
            9.33241349802137354352282727631973e-01 2.07245322480992089086981877699145e-01 -1.40486672283129443439264605331118e-01 2.22712740876849162469837010774540e-08
            7.18814607014818252039844992395956e-01 -1.08592374847109193747307642752276e-01 3.89777767832290955585250458170776e-01 1.36513865091716159105977575032032e-07
            -7.53425185865285035324845352988632e-02 3.65738826101051017758436501026154e-01 7.09603692485477388629533379571512e-01 1.26203234085911475754928755960149e-06
            -7.29720466214083957057212614927266e-02 9.08780987704670639715232027810998e-01 1.64191058916737686601550194609445e-01 1.27738868592075040974331487375615e-06
            1.61390053532476512865301288002229e-01 -5.67779747516548999319851986911090e-02 8.95387921219178428700047334132250e-01 3.45155017609913357620765074484925e-06
            3.96473037509524617316714056869387e-01 -3.93347505474984995843534818504850e-02 6.42861713037973792062018674187129e-01 1.72714542050300065036388258432609e-05
            3.25670755950673990186672313029703e-02 9.83615991855380511488249339890899e-01 -1.61830674504479521402799946372397e-02 1.01025648903933112400521654450358e-04
            5.11083489768881849002468698017765e-01 -1.67145669135699004304029813283705e-02 5.05631077144688023672358667681692e-01 1.02898957150498768600178911558629e-04
            9.69815309580979389991739481047262e-01 -8.47386414977770001388801546227114e-03 3.86585545687983117568720103918167e-02 2.58497874271555380925463607511006e-04
            4.56684802229330010642227577477570e-03 1.12732881898779998403048452360053e-02 9.84159863787828603776119962276425e-01 6.92624886129114070501544553337681e-04
            9.78534773184665751522004484286299e-01 1.31017828795071001463101723061300e-02 8.36344393582714833168534340757105e-03 1.41412835166943289672003558621327e-03
            -2.09366083091479986480476505050774e-03 6.11172971166304646040146053564968e-01 3.90920689664610132574296130769653e-01 1.49301067145266920001278609930750e-03
            8.24465917969609968229960372809728e-03 9.77374814635139754770420950080734e-01 1.43805261851641663639611579128541e-02 1.51297790193467211576627828861774e-03
            2.86161958663017010340290369185823e-02 8.39418112704680072222451769903273e-03 9.62989623006651473957617781707086e-01 1.70688117637062680698178951388400e-03
            6.57626401564029988699999762502557e-03 5.28100601072915010192332374572288e-02 9.40613675877068233788236284453887e-01 1.93499206216790555108719917853932e-03
            2.75866644961711204864229785016505e-01 4.21287699982659993164224587758326e-03 7.19920478038462152703402807674138e-01 2.50882945933416116532366224589623e-03
            1.05868943093978096214868855895475e-01 8.86885891425699846024599537486210e-01 7.24516548032205776053160661831498e-03 3.08849742121904830005618691757263e-03
            6.19226480612458551888721558498219e-01 3.76941832682914701457832507003332e-01 3.83168670462674665344593449844979e-03 3.10612589259128078073013412563341e-03
            8.83776214104550317252062541228952e-01 8.37053232048790007979732763487846e-03 1.07853253574961782668140131136170e-01 3.50615202853169459495941140403374e-03
            9.25410583696847699464171910221921e-01 6.44641060621398004348847621258756e-02 1.01253102410125001009433276522032e-02 3.55650926910889035245655342976079e-03
            1.08650060510514998024511612584320e-02 9.20664143492036624039087655546609e-01 6.84708504569119247307185105455574e-02 3.94966708004545249272032592102732e-03
            9.44141087572360031276996750193575e-03 1.22132342422849995977607306940627e-01 8.68426246701426429730474865209544e-01 4.03761777950527919051060266042441e-03
            8.34379668080313940947689843596891e-02 1.14591516812569006039668195739978e-02 9.05102881510711720913775479857577e-01 4.09379962800071953155089943265921e-03
            4.61305836421291026727686812591855e-02 9.33288093591807421667283506394597e-01 2.05813227660635034155234279751312e-02 4.73403115822502475185107329025413e-03
            9.30132604326048717524599851458333e-01 2.32412078745275994806629427102962e-02 4.66261877994236795252902538777562e-02 4.95930153832710007150152975441415e-03
            4.00920237531567025257572822738439e-01 5.85675939955597968378242512699217e-01 1.34038225128350063641846645623446e-02 5.08410509645927655153485602568253e-03
            2.00298631844479407826398187353334e-01 7.90161352541371209134979380905861e-01 9.54001561414941079419804736971855e-03 5.15375868661858103914052975369486e-03
            3.77632914144658965449607990194636e-02 4.88000038207256972566661090695561e-02 9.13436704764808427015054803632665e-01 5.41753935016131039914544231805849e-03
            7.86993303590034054550983455555979e-01 1.02546720868124992959158348071469e-02 2.02752024323153451357271137567295e-01 5.66845217213470670136743478906283e-03
            8.40737863810381957918593798240181e-01 1.47322288830840891771245537711366e-01 1.19398473587771503101606640484533e-02 5.83413421137796671389619262981796e-03
            1.05197174565543997981631463289887e-02 2.19035042530801610238810894770722e-01 7.70445240012643917104639967874391e-01 5.89591822146309871982561290337799e-03
            9.97901146028259966591633656207705e-03 7.27289788380431234671164020255674e-01 2.62731200159286193418495258811163e-01 5.90289996316258579189817368160220e-03
            3.10283140029019188776970850085490e-01 6.78994359625208487862835227133473e-01 1.07225003457723788713451540388633e-02 5.98410059681952089666312488702715e-03
            4.90189560074439811820923296181718e-01 4.99174527028812708984162327396916e-01 1.06359128967474791949143764213659e-02 6.07664925589345393530660288661238e-03
            1.67966330536045205246153955158661e-01 1.41053909355621006488368252007604e-02 8.17928278528392671553604031942086e-01 6.80802987370580838660805156337119e-03
            1.41127681947915001314797223130881e-02 8.32447463133275222801898962643463e-01 1.53439768671933318699984738486819e-01 6.91847930674361916852932452570712e-03
            1.04234395787746996625555695459298e-02 4.68998077755451825421317835207446e-01 5.20578482665773556448129966156557e-01 7.10325605172282877747269580481770e-03
            7.33146336217226024523085925466148e-01 2.54442450218429205843762019867427e-01 1.24112135643447696331520546664251e-02 7.33463576813609199889221201829059e-03
            2.76872655309121973310482189845061e-01 2.38950051964319995501107030122512e-02 6.99232339494446075711664434493287e-01 7.43751327716430862824203984473570e-03
            6.67893530919515643340389488002984e-01 1.20851625782620996502059185218059e-02 3.20021306502222269152468925312860e-01 7.68365168139582832562695102751604e-03
            4.45015015162890034683407236570929e-02 6.96409123129440255084432465082500e-01 2.59089375354270790019484138611006e-01 8.52988989386579445073355287831873e-03
            1.37349285317673996259513202744529e-02 3.36939397252163819196368876873748e-01 6.49325674216068793320744134689448e-01 8.66606394843610731981708994453584e-03
            5.37547879908140235905023018858628e-01 1.38730562867831995477185813570031e-02 4.48579063805076561077811447830754e-01 8.89049133285853970809498036942387e-03
            5.92835053410133872020537637581583e-01 3.77159114650097326837396849441575e-01 3.00058319397688011420655129768420e-02 9.43519996936341073801024492695433e-03
            1.26630109374717497727758086512040e-01 8.30222658453748652007675445929635e-01 4.31472321715338225089908519294113e-02 9.44330294935880401596683952902822e-03
            4.01366382406037780494756361804320e-01 1.50259944876660004459134611920490e-02 5.83607623106296213855159749073209e-01 9.55587876959992921221953565691365e-03
            8.57954275672313304568206149269827e-01 8.62277341830808979894484878059302e-02 5.58179901446057974423453629242431e-02 9.76792576902996145804536354262382e-03
            4.91710116722589990478908816839976e-02 1.23321540053355493959941213688580e-01 8.27507448274385493114380096812965e-01 9.79368993772792260921189466671422e-03
            5.90361244964826975545335585593421e-02 8.59769883879450080854667248786427e-01 8.11939916240672632241626160976011e-02 9.84501672135575447986077790574200e-03
            8.28552704610202916946093409933383e-01 4.72711454279717974147345671553921e-02 1.24176149961825285639172022911225e-01 9.89476521934182402795432409448040e-03
            1.02888155475812897998721950898471e-01 6.05783796272693006446097285788710e-02 8.36533464896917822173350032244343e-01 1.04456538049990978189729418090792e-02
            2.52668780123645984880820236639920e-02 5.90679259483232632632621061929967e-01 3.84053862504402765409849962452427e-01 1.15273909314429125888556981749389e-02
            5.22224101343283023113350793664722e-02 4.46404157593296102568558580969693e-01 5.01373432272375629814575859199977e-01 1.27672922531254316030890905153683e-02
            5.57922911259374973425018140460452e-02 2.21958351229682604444093385609449e-01 7.22249357644379808007784049550537e-01 1.34789031134591053701088370075922e-02
            2.33057059900132507168635243033350e-01 7.14868759196109149200992760597728e-01 5.20741809037583713859476119978353e-02 1.35880267869236126543341924843844e-02
            7.60238740904368159867487975134281e-01 1.78054468985292091431915650900919e-01 6.17067901103397487005963739647996e-02 1.36728497497001257043880428909688e-02
            6.53782134850099616585339390439913e-01 2.84349895545913999850284881176776e-01 6.18679696039863835643757283833111e-02 1.38381578396779940376415396485754e-02
            7.01192668918892009211063509610540e-02 7.61645772034410195594489323411835e-01 1.68234961073700617362192133441567e-01 1.39248345006680734892157147442049e-02
            8.56752082434592970860975924551894e-02 6.20123951286305374885898800130235e-01 2.94200840470235314150215799600119e-01 1.39770088099752387567065170514979e-02
            7.17456088900951804987471405183896e-01 5.55931048067861977024506359157385e-02 2.26950806292261997310077958900365e-01 1.41237710737246464215433050526372e-02
            1.97548193096344609775982803512306e-01 6.85029623840039958437841960403603e-02 7.33948844519651366624657384818420e-01 1.47986633175174308457711092046338e-02
            8.64145042160261028785228631932114e-02 5.13438219270665352489402266655816e-01 4.00147276513308502998711446707603e-01 1.52912143289120460581953508949482e-02
            4.75509760996478514361029965584748e-01 4.64836327100429602676712192987907e-01 5.96539119030919384734090726851719e-02 1.55539999414216756346007741740323e-02
            6.78714687817086048671200160242734e-02 3.31996502196124476657956847702735e-01 6.00132029022167001741649983159732e-01 1.58026812121302914770915037934174e-02
            1.46979207784701609851296666420239e-01 7.41078729902387989270096113614272e-01 1.11942062312910373123031604336575e-01 1.62346086473484640944064949508174e-02
            3.51221665184432485506960119892028e-01 5.87953585294961000862201672134688e-01 6.08247495206065691419894392311107e-02 1.62579309869656761511347298210239e-02
            7.41261492762090412611541978549212e-01 1.20894088556726203576019429419830e-01 1.37844418681183383812438592030958e-01 1.66874062944868742364157299107319e-02
            1.24040327659173499341704882681370e-01 1.42583601364923207022172846336616e-01 7.33376070975903293636122270982014e-01 1.67111604129371975335782707361432e-02
            2.13880394121143901919879226625198e-01 2.70490112816105798820842665008968e-01 5.15629493062750299259278108365834e-01 1.72247776299753949102999683873350e-02
            5.88770407739171108829623335623182e-01 6.28867829706775960385556345499936e-02 3.48342809290151267376245414197911e-01 1.72494447262747374605140748826670e-02
            3.21270098954109073474683100357652e-01 7.16976174401817989423335575338569e-02 6.07032283605709155338558957737405e-01 1.72772509570036612858157099026357e-02
            4.54559036809930927525158494972857e-01 6.48529251240834991865114034226281e-02 4.80588038065985601043905717233429e-01 1.78662725887897728960229670747140e-02
            2.88845281940968090506061116684577e-01 2.38118396866765308361735264952586e-01 4.73036321192266573376628002733923e-01 1.80101855714390364504584596261338e-02
            5.32163580900152477504150283493800e-01 3.57017738203314505263108458166244e-01 1.10818680896533017232741258339956e-01 1.87684488537231825011630803601292e-02
            4.73641029272273195172004989217385e-01 3.35467115867110698967934467873420e-01 1.90891854860616050348909311651369e-01 1.93795009936486538337785390240242e-02
            1.37215640807511696230847064725822e-01 2.43905972104074802997430992945738e-01 6.18878387088413473016146326699527e-01 1.98626606272718116141806632413136e-02
            6.21391658733018270588388531905366e-01 1.35037461036809303127625980778248e-01 2.43570880230172426283985487316386e-01 2.12946646299398724411222616481609e-02
            1.52603655913764513218922047599335e-01 6.34085174784072780340693498146720e-01 2.13311169302162761951535685511772e-01 2.15269076503021503099510880474554e-02
            2.53542499404753318703598097272334e-01 6.14568220520721020250221044989303e-01 1.31889280074525716557332088996191e-01 2.16331030754623523937318907428562e-02
            6.30459386519128472947670616122195e-01 2.25032882061072608825824659106729e-01 1.44507731419798918226504724771075e-01 2.22370855987509032092308558503646e-02
            2.30848514163360801765279006758647e-01 1.50778181472718098010687981513911e-01 6.18373304363921016957306164840702e-01 2.26361907194113400054735762978453e-02
            3.60607666765002221609393018297851e-01 1.47657125036602110412786714732647e-01 4.91735208198395667977820266969502e-01 2.29210101918507595797347420329970e-02
            1.68825843777590689498424580961000e-01 4.96801767224627877084230931359343e-01 3.34372388997781433417344487679657e-01 2.33618251134492804776954244516673e-02
            5.12950485421223789650468916079262e-01 2.34938455462988304844884623889811e-01 2.52111059115787905504646460030926e-01 2.36544314840157199331116544271936e-02
            4.87479899308838782179265081140329e-01 1.44753653681138988451593263562245e-01 3.67766447010022257124717270926340e-01 2.41824332743506009713030380225973e-02
            1.41411858661172812334783088772383e-01 3.79555454421125593178487633849727e-01 4.79032686917701622242304893006803e-01 2.43357949855367329539124199300204e-02
            3.79920224601669520581026517902501e-01 4.77536434179863300553137150927796e-01 1.42543341218467178865836331169703e-01 2.53763936634402234582275070806645e-02
            2.59931264032647602046921520013711e-01 5.04552912634035233274687470839126e-01 2.35515823333317109167239777889336e-01 2.56678765096296571113221318682918e-02
            3.60690792192417675199322957269032e-01 3.78196347277973687894814247556496e-01 2.61112860529608692417014026432298e-01 2.59481828761645126613100131862666e-02
            3.85985301830885418894467875361443e-01 2.56761175269934582843234238680452e-01 3.57253522899179998262297885958105e-01 2.76370050143116899321604762462812e-02
            2.54967729527999675820382208257797e-01 3.69733046040209401006393363786628e-01 3.75299224431790867662073196697747e-01 2.83606567008948921848610069673668e-02
            ];
        
        
    otherwise
        xyw=[];
end




pointset_stats=[
    11   26 4.72e-16    0    1
    20   77 1.45e-15    0    8
    22   92 1.72e-15    0   11
    ];





