
function [xyw_bar,pointset_stats]=set_taylor_wingate_bos_C_barycentric(deg)

%--------------------------------------------------------------------
% CUBATURE RULE ON THE SIMPLEX [0 0; 0 1; 1 0] WITH ADE EQUAL TO deg.
% THE CUBATURE RULE IS DESCRIBED IN BARYCENTRICAL FORM.
%--------------------------------------------------------------------
% INPUT:
% deg: algebraic degree of precision (ADE) of the rule.
%--------------------------------------------------------------------
% OUTPUT:
% xyw_bar: MATRIX OF NODES xyw(:,1:3) IN BARYCENTRICAL COOR. AND WEIGHTS xyw(:,4).
%      REMARK: The sum of the weights makes 1.
%
% pointset_stats:
%    statistics of the pointset stored in a matrix. Each rows contains
%    degree, cardinality of the full pointset, moment error w.r.t.
%    orthn. Dubiner basis, number of points with negative weights,
%    number of points outside the domain.
%--------------------------------------------------------------------
% SOURCE:
%  Mark A. Taylor, Beth A. Wingate, Len P. Bos,
%  "Several new quadrature formulas for polynomial integration in the triangle",
%  unpublished note, see arXiv.
%--------------------------------------------------------------------
% | DEG | CARD | W- | PT.O|   M.E.   |Q.|
%--------------------------------------------------------------------
% |   2 |   3 |   0 |   0 | 2.12e-16 |PI|
% |   4 |   6 |   0 |   0 | 2.22e-16 |PI|
% |   5 |  10 |   0 |   0 | 1.77e-13 |PI|
% |   7 |  15 |   0 |   0 | 4.59e-13 |PI|
% |   9 |  21 |   0 |   0 | 3.45e-16 |PI|
% |  11 |  28 |   0 |   0 | 4.34e-16 |PI|
% |  13 |  36 |   0 |   0 | 5.46e-16 |PI|
% |  14 |  45 |   0 |   0 | 4.53e-13 |PI|
% |  16 |  55 |   0 |   0 | 7.82e-13 |PI|
% |  18 |  66 |   0 |   0 | 4.77e-16 |PI|
% |  20 |  78 |   0 |   0 | 6.04e-16 |PI|
% |  21 |  91 |   0 |   0 | 5.97e-16 |PI|
% |  23 | 105 |   0 |   0 | 8.74e-16 |PI|
% |  25 | 120 |   0 |   3 | 6.97e-16 |P0|
%--------------------------------------------------------------------
% LEGEND:
%--------------------------------------------------------------------
% DEG  : algebraic degree of precision
% CARD : cardinality of the pointset
% W-   : negative weights
% PT.O : points outside the simplex [0 0; 1 0; 0 1].
% M.E. : maximum moment error in inf norm.
% Q.: quality of the rule.
%        PI: pos. weights, points in simplex
%        NI: some neg. weights, points in simplex
%        P0: pos. weights, some points out of the simplex
%        N0: some neg. weights, some points out of the simplex
%--------------------------------------------------------------------



switch deg
    
    case   2
        
        % ALG. DEG.:    2
        % PTS CARD.:    3
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 2.1e-16
        
        xyw_bar=[
            1.66666666666666712925959359381523e-01 6.66666666666699936349971267190995e-01 1.66666666666633322968493757798569e-01 3.33333333333288961419782481243601e-01
            6.66666666666700047372273729706649e-01 1.66666666666666601903656896865868e-01 1.66666666666633350724069373427483e-01 3.33333333333288850397480018727947e-01
            1.66666666666699964105546882819908e-01 1.66666666666699964105546882819908e-01 6.66666666666600127300057465618011e-01 3.33333333333422243693888731286279e-01
            ];
        
    case   4
        
        % ALG. DEG.:    4
        % PTS CARD.:    6
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 2.2e-16
        
        xyw_bar=[
            9.15762135097877988387793379843060e-02 9.15762135097891449841966959866113e-02 8.16847572980422986788084926956799e-01 1.09951743655353692541432053531025e-01
            8.16847572980492264704821536724921e-01 9.15762135097538954031648472664529e-02 9.15762135097538398920136160086258e-02 1.09951743655291589440992083837045e-01
            9.15762135097846763365225797315361e-02 8.16847572980460179259409869700903e-01 9.15762135097551999152187818253879e-02 1.09951743655320358095117683205899e-01
            1.08103018168073619631464055146353e-01 4.45948490915999973260852584644454e-01 4.45948490915926365474319936765824e-01 2.23381589678016162503482178181002e-01
            4.45948490915999973260852584644454e-01 1.08103018168070386106904834377929e-01 4.45948490915929640632242580977618e-01 2.23381589678011693855808061925927e-01
            4.45948490915999973260852584644454e-01 4.45948490915933359879375075252028e-01 1.08103018168066666859772340103518e-01 2.23381589678006559074319170576928e-01
            ];
        
    case   5
        
        % ALG. DEG.:    5
        % PTS CARD.:   10
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.8e-13
        
        xyw_bar=[
            0.00000000000000000000000000000000e+00 1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 1.31356049751999992558637586625991e-02
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 1.31358306033999994860694826570580e-02
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 1.00000000000000000000000000000000e+00 1.37081973800000003066434928200579e-02
            2.67327353118499977391309130325681e-01 6.72819921871000037327803511288948e-01 5.98527250104999852808873583853710e-02 1.17419193291149998037425916663778e-01
            6.72817552946100017052799557859544e-01 2.67328859948199992757622567296494e-01 5.98535871056999901895778748439625e-02 1.17420611913400002590535109447956e-01
            6.49236350053999933251702714187559e-02 6.71653011149400014367927269631764e-01 2.63423353845199992306902458949480e-01 1.24012589655700000701443741490948e-01
            6.71649853904199956033949092670809e-01 6.49251690029000044779294853469764e-02 2.63424977092900025610333614167757e-01 1.24015246126049999420004610328760e-01
            6.54032456799999994956351656583138e-02 2.69378936645299993379865099996096e-01 6.65217817674700007124499734345591e-01 1.25930230276450005888477789994795e-01
            2.69376706914000008108445172183565e-01 6.54054874919000051036732656939421e-02 6.65217805594099931276730330864666e-01 1.25933026682899990023045688758430e-01
            3.38673850389600017418700872440240e-01 3.38679989302699990805223251300049e-01 3.22646160307699936264924645001884e-01 2.25289469095699995682124949780700e-01
            ];
        
    case   7
        
        % ALG. DEG.:    7
        % PTS CARD.:   15
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.6e-13
        
        xyw_bar=[
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 5.12790870460000033914660377831751e-03
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 1.00000000000000000000000000000000e+00 5.12790870460000033914660377831751e-03
            0.00000000000000000000000000000000e+00 1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 5.12790870460000033914660377831751e-03
            7.83965665101200026576577783998800e-01 4.21382841641999988935296528325125e-02 1.73896050734599960652104755354230e-01 5.58023523323500011961151301420614e-02
            1.73896050734500012824312875636679e-01 7.83965665101200026576577783998800e-01 4.21382841642999883546849559934344e-02 5.58023523323500011961151301420614e-02
            1.73896050734500012824312875636679e-01 4.21382841641999988935296528325125e-02 7.83965665101300057671096510603093e-01 5.58023523323500011961151301420614e-02
            4.21382841641999988935296528325125e-02 1.73896050734500012824312875636679e-01 7.83965665101300057671096510603093e-01 5.58023523323500011961151301420614e-02
            7.83965665101200026576577783998800e-01 1.73896050734500012824312875636679e-01 4.21382841642999605991093403645209e-02 5.58023523323500011961151301420614e-02
            4.21382841641999988935296528325125e-02 7.83965665101200026576577783998800e-01 1.73896050734600016163255986612057e-01 5.58023523323500011961151301420614e-02
            4.74388086175200007765795362502104e-01 4.74388086175200007765795362502104e-01 5.12238276495999844684092749957927e-02 8.39887797667499963472081958570925e-02
            4.74388086175200007765795362502104e-01 5.12238276497000016851401937856281e-02 4.74388086175099976671276635897812e-01 8.39887797667499963472081958570925e-02
            5.12238276497000016851401937856281e-02 4.74388086175200007765795362502104e-01 4.74388086175099976671276635897812e-01 8.39887797667499963472081958570925e-02
            2.38561530018099987593416244635591e-01 5.22876939963899944885383774817456e-01 2.38561530018000067521199980546953e-01 1.32611940197299993382529237351264e-01
            5.22876939963899944885383774817456e-01 2.38561530018099987593416244635591e-01 2.38561530018000067521199980546953e-01 1.32611940197299993382529237351264e-01
            2.38561530018099987593416244635591e-01 2.38561530018099987593416244635591e-01 5.22876939963800024813167510728817e-01 1.32611940197299993382529237351264e-01
            ];
        
    case   9
        
        % ALG. DEG.:    9
        % PTS CARD.:   21
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.5e-16
        
        xyw_bar=[
            4.51890097843806529120946890998312e-02 4.51890097843734711569041451184603e-02 9.09621980431245868992107261874480e-01 2.59935710323200112881192325176016e-02
            4.51890097843737903460237248509657e-02 9.09621980431245868992107261874480e-01 4.51890097843803406618690132745542e-02 2.59935710323200459825887520537435e-02
            9.09621980431248200460458974703215e-01 4.51890097843794594223432170565502e-02 4.51890097843723401171978082402347e-02 2.59935710323187137149592018658950e-02
            7.47512472733889321574451969354413e-01 3.04243617288123165132152081469030e-02 2.22063165537298351503991966637841e-01 3.53517050891897222086868168844376e-02
            2.22063165537333934151931202904962e-01 3.04243617288317072522296768966044e-02 7.47512472733834365534733024105662e-01 3.53517050892123221861318427272636e-02
            7.47512472733836363936177349387435e-01 2.22063165537328827126017927184876e-01 3.04243617288348089378047234276892e-02 3.53517050892150699881177899897011e-02
            2.22063165537305123864442180092738e-01 7.47512472733883548414723918540403e-01 3.04243617288113554764095169957727e-02 3.53517050891900413978063966169429e-02
            3.04243617288244561081000938429497e-02 7.47512472733840471761368462466635e-01 2.22063165537335072130531443690415e-01 3.53517050892048836918668541784427e-02
            3.04243617288046282187696789378606e-02 2.22063165537305429175773952010786e-01 7.47512472733889987708266744448338e-01 3.53517050891829914816000268729113e-02
            1.36991201264899997713797574760974e-01 2.18290070971362881735444716468919e-01 6.44718727763737176061908940027934e-01 4.54695380476657778912041862895421e-02
            6.44718727763699983590583997283829e-01 2.18290070971372207608851567783859e-01 1.36991201264927808800564434932312e-01 4.54695380475926697050326197313552e-02
            1.36991201264899997713797574760974e-01 6.44718727763739174463353265309706e-01 2.18290070971360883334000391187146e-01 4.54695380476010241332929240343219e-02
            2.18290070971400934629613743709342e-01 6.44718727763699983590583997283829e-01 1.36991201264899054024226643377915e-01 4.54695380476398750002609006060084e-02
            2.18290070971399990940042812326283e-01 1.36991201264899997713797574760974e-01 6.44718727763700094612886459799483e-01 4.54695380475752738980155243098125e-02
            6.44718727763712418088459799037082e-01 1.36991201264899997713797574760974e-01 2.18290070971387584197742626201943e-01 4.54695380476408672620891593396664e-02
            3.69603304333801299663697648156813e-02 4.81519834783289857327304162026849e-01 4.81519834783330047400795592693612e-01 5.16172025690235861117294291489088e-02
            4.81519834783337374872758118726779e-01 3.69603304333760152022847478292533e-02 4.81519834783286637680532749072881e-01 5.16172025690189509306016191203526e-02
            4.81519834783297573377325306864805e-01 4.81519834783325717530999554583104e-01 3.69603304333766535805239072942641e-02 5.16172025690197142089310489154741e-02
            4.03603979817900027082799851996242e-01 1.92792040364117206463490106216341e-01 4.03603979817982683186983194900677e-01 9.40800734583555342549487932046759e-02
            4.03603979817979185984455625657574e-01 4.03603979817903135707268802434555e-01 1.92792040364117733819426803165697e-01 9.40800734583540076982899336144328e-02
            1.92792040364124867002360019796470e-01 4.03603979817978575361792081821477e-01 4.03603979817896529880272282753140e-01 9.40800734583589065573860921176674e-02
            ];
        
    case  11
        
        % ALG. DEG.:   11
        % PTS CARD.:   28
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.3e-16
        
        xyw_bar=[
            2.11168130454084196363955981711748e-14 9.45170445017362803596938647388015e-01 5.48295549826161021655934746377170e-02 5.70412470163417055502019792356805e-03
            9.45170445017299964973744863527827e-01 4.90750005846733871211117897806156e-14 5.48295549826509631685667045530863e-02 5.70412470164347734646881349362957e-03
            9.28900240571878477169320831308141e-01 6.85505797224144203250162377116794e-02 2.54917970570710250566293098017923e-03 6.63456428597094041710313661042164e-03
            6.85505797224479906937233408825705e-02 9.28900240571737367822890973911854e-01 2.54917970581469699453691646340303e-03 6.63456428600110118681820736696864e-03
            2.43268355615265073621511504597947e-02 2.43268355615932664604006419040161e-02 9.51346328876880198421872592007276e-01 7.79328866749673648123275526700127e-03
            1.27966283533483604628600005526096e-01 2.77838749488136006748906225993778e-02 8.44249841517702770410380708199227e-01 2.04137390214265204702037692641170e-02
            2.77838749488144090560304277914838e-02 1.27966283533699987096099448535824e-01 8.44249841517485610786764027579920e-01 2.04137390214421260425936566207383e-02
            2.87083428360431214265702237753430e-02 7.49834758865739803290750842279522e-01 2.21456898298217130793830165202962e-01 2.89924832557993984727406910906211e-02
            7.49834758865634110058806527376873e-01 2.87083428360427571346402686458532e-02 2.21456898298323129337106252023659e-01 2.89924832558150977201982811948255e-02
            7.22800790970692719206169840617804e-01 2.49760206238569426595574896055041e-01 2.74390027907378541982552633271553e-02 3.00692623831495010888747998478721e-02
            2.49760206238600013239903319117730e-01 7.22800790970658635359313848312013e-01 2.74390027907414069119340638280846e-02 3.00692623831564087577561394937220e-02
            8.65562992839277733914116197411204e-02 8.32551385699644819204934265144402e-01 8.08923150164274629148053463723045e-02 3.12636944216442974608938243363809e-02
            8.32551385699719315169886613148265e-01 8.65562992839324085725394297696766e-02 8.08923150163482762575739570820588e-02 3.12636944216627341019965058421803e-02
            3.06161915767199976912138481566217e-01 3.03526617490630160056674924362596e-02 6.63485422483736986265512314275838e-01 3.19842160751948184627124760481820e-02
            3.03526617490652954323149259607817e-02 3.06161915767456771497734280274017e-01 6.63485422483477971233867265254958e-01 3.19842160751945270291685119445901e-02
            4.86861059504704873912572793415166e-01 4.86861059504661464192309949794435e-01 2.62778809906336063839660255325725e-02 3.30662936080424879614625410795270e-02
            6.65790429301691855101807959727012e-01 1.76545615421862284177834112597338e-01 1.57663955276445860720357927675650e-01 3.34251618410003892289950044869329e-02
            1.76545615422072060818337035925651e-01 6.65790429301600039657671459281119e-01 1.57663955276327927279567120422143e-01 3.34251618410489545474284511783480e-02
            2.93121007359746013620771520891140e-02 5.29565748866929753013721438037464e-01 4.41122150397095680318670929409564e-01 3.43452152988478615069922739166941e-02
            5.29565748866726582200215034390567e-01 2.93121007359754374987925729101335e-02 4.41122150397297962953757632931229e-01 3.43452152988611564277121601662657e-02
            1.44467382439143698924510772485519e-01 1.44467382439153968487488555183518e-01 7.11065235121702388099151903588790e-01 5.01358771929493740593386519321939e-02
            3.29974011141099976551771533195279e-01 5.36181572904960379588601426803507e-01 1.33844415953939588348475808743387e-01 5.71568392049582527092432826520962e-02
            5.36181572905225722891486839216668e-01 3.29974011140833856092768883172539e-01 1.33844415953940421015744277610793e-01 5.71568392049688484002345489898289e-02
            5.51150751686180218058552782167681e-01 1.43779086192284799583518406507210e-01 3.05070162121534982357928811325110e-01 6.11824073375901028581935747752141e-02
            1.43779086192285576739635644116788e-01 5.51150751686222073466581150569255e-01 3.05070162121492294282631974056130e-01 6.11824073376113913846907621518767e-02
            3.34806658732700601621701252952334e-01 1.52961943716094977974151447597251e-01 5.12231397551204503670874146337155e-01 6.97211167088987310291869903267070e-02
            1.52961943716101833601328507938888e-01 3.34806658732712258963459817096009e-01 5.12231397551185851924060443707276e-01 6.97211167089028249765902955914498e-02
            3.43018349814737999459168804605724e-01 3.43018349814726564162015165493358e-01 3.13963300370535436378816029900918e-01 8.72188914590879132981982024830359e-02
            ];
        
    case  13
        
        % ALG. DEG.:   13
        % PTS CARD.:   36
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.5e-16
        
        xyw_bar=[
            2.42935351590266995769340496735822e-02 9.49305929384640645452009266591631e-01 2.64005354563326966044201071781572e-02 8.31204993785482869339542588704717e-03
            2.65193427721596534929471289387948e-02 2.42695130640411434186365369214400e-02 9.49211144163799258599567565397592e-01 8.34058498887744505234653757952401e-03
            9.49212602355106738194479021331063e-01 2.65067966437244201260980958068103e-02 2.42806010011688416794228828621272e-02 8.34152845332809290967990278886646e-03
            3.37757637490039737404590169944640e-03 4.76731641236311620701826541335322e-01 5.19890782388787942025487609498668e-01 8.78404350412627377142626983186346e-03
            4.75767229810120817479202059985255e-01 5.19892182910199562861919275746914e-01 4.34058727967967517002989552565850e-03 9.22373309223941360845966386250439e-03
            5.19078319347068495304142743407283e-01 5.59127062021505007172184065211695e-03 4.75330410032716477175540603639092e-01 9.89712050939165963181487484234822e-03
            8.61683974532058805095857678679749e-01 1.33996048618149239672892036878693e-02 1.24916420606126274406300069585996e-01 1.01770197927333813647798876900197e-02
            1.24920975992555027822561442008009e-01 8.61305432133411819073387505341088e-01 1.37735918740331531040510526509024e-02 1.03426431970092581880127369231559e-02
            1.38565453861037562316438354059756e-02 1.24773371735846919095003215716133e-01 8.61370082878049392327568511973368e-01 1.04135683043232105116082664153510e-02
            2.11887064221680694597882421703616e-02 8.43843835122266905379717627511127e-01 1.34967458455565014752153274457669e-01 1.58909889139352972220464721431199e-02
            8.43229678721886921621830879303161e-01 1.35456364583034505333358765710727e-01 2.13139566950785730448103549861116e-02 1.60236017620355025714573571349320e-02
            1.35423179786502867472108846413903e-01 2.13482820656213274912449406883752e-02 8.43228538147875794628305357036879e-01 1.60303840572841617417854109817199e-02
            3.08885351067946345438031130470335e-01 2.21919663013589105182710170538485e-02 6.68922682630694764860379564197501e-01 2.15382979591376561012694423880021e-02
            6.68505759516906050698992203251692e-01 3.08901287938941060406961014450644e-01 2.25929525441528888940467822976643e-02 2.19236707669562061462364965791494e-02
            2.26545012557140094944241326402334e-02 6.69170994332103252766330570011633e-01 3.08174504412182748147586153208977e-01 2.19604836366373672162932706442007e-02
            2.80851540877201766299720020469977e-01 6.92471815510625865464078287914163e-01 2.66766436121723682362016916158609e-02 2.39975961845282532991507906672268e-02
            6.92244674905063694936302454152610e-01 2.68723345025942798869333927314074e-02 2.80882990592342007829529393347912e-01 2.41903130366623933533709589482896e-02
            2.68617447119446836611977857955935e-02 2.81009397322192155943554325858713e-01 6.92128857965863097945202753180638e-01 2.42433711687330862016853672002981e-02
            1.14177848547014335056637435172888e-01 7.97358141358575411850040381978033e-01 8.84640100944102947266856062924489e-02 2.78482244012102607422232125600203e-02
            7.97480792206136324118403990723891e-01 8.79806508790785812834656098857522e-02 1.14538556914785094598130399390357e-01 2.80513182178032290103875112663445e-02
            8.92807293894229275776908139050647e-02 1.14502056112754088679217545632127e-01 7.96217214497823011498667256091721e-01 2.82595061846439496444638450611819e-02
            1.05248789245499116407067674572318e-01 6.68690411992205024915847388911061e-01 2.26060798762295900310448359959992e-01 3.44644945334759839061788966319000e-02
            6.66302228074011737568582702806452e-01 2.27505163183173791940916430576181e-01 1.06192608742814470490500866617367e-01 3.58606668044348514401420402464282e-02
            2.30780373754708584277395289063861e-01 1.05457256122134482301255786751426e-01 6.63762370123156864032409885112429e-01 3.63726960488190351195036953413364e-02
            1.70505915754032177922283608495491e-01 5.17406439865800038546694850083441e-01 3.12087644380167783531021541421069e-01 3.94403668368502255403562628544023e-02
            5.08659397304299987219167178409407e-01 3.17052385520937729523893722216599e-01 1.74288217174762283256939099373994e-01 4.05057172756123629042690481583122e-02
            3.14182386228100019032183354283916e-01 1.81070636165945320072978574899025e-01 5.04746977605954660894838070817059e-01 4.12862649527448860586886780765781e-02
            4.61746081786397921664644172778935e-01 4.67859453980425554764366324889124e-01 7.03944642331765790821407335897675e-02 4.21022283665200280911022900909302e-02
            6.93087496081039450279703828528000e-02 4.62285604208458600350439837711747e-01 4.68405646183437440743801971620996e-01 4.21792766652516123992988639201940e-02
            4.65195525926812847306024423232884e-01 7.24357805669117588154293230218173e-02 4.62368693506275463267485292817582e-01 4.25984934243976154366606579060317e-02
            2.57862585789255749357806735133636e-01 6.13139503917723982517884451226564e-01 1.28997910293020212613157582381973e-01 4.51422664026207634546672409214807e-02
            6.11262776677922392565278641995974e-01 1.30036083460934759337135346868308e-01 2.58701139861142848097586011135718e-01 4.57141571742254032573171684816771e-02
            1.30518213593359272417515626329987e-01 2.58171382888366274244162923423573e-01 6.11310403518274481093897065875353e-01 4.58139532704636143201071263320046e-02
            4.28143799182839734918104568350827e-01 2.36200596981703009324959907644370e-01 3.35655603835457228001359908375889e-01 5.12786687447955580321234947405173e-02
            3.35699578373029994260434705211082e-01 4.31102630858835600413669908448355e-01 2.33197790768134405325895386340562e-01 5.16579830706739157042406418440805e-02
            2.30542429883610316965203423933417e-01 3.45601394937618655589517402404454e-01 4.23856175178771055200854789291043e-01 5.17927183596631687811928657083627e-02
            ];
        
    case  14
        
        % ALG. DEG.:   14
        % PTS CARD.:   45
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.5e-13
        
        xyw_bar=[
            0.00000000000000000000000000000000e+00 1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 5.30835599499999998863997152653837e-04
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 5.30835599499999998863997152653837e-04
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 1.00000000000000000000000000000000e+00 5.30835599499999998863997152653837e-04
            5.73330873025999984693434896598774e-02 1.51382269814000006147303167836071e-02 9.27528685716000000915926193556515e-01 6.57301180504999988524206244733250e-03
            5.73330873025999984693434896598774e-02 9.27528685716000000915926193556515e-01 1.51382269813999936758364128763787e-02 6.57301180504999988524206244733250e-03
            9.27528685716000000915926193556515e-01 5.73330873025999984693434896598774e-02 1.51382269814000006147303167836071e-02 6.57301180504999988524206244733250e-03
            1.51382269814000006147303167836071e-02 5.73330873025999984693434896598774e-02 9.27528685716000000915926193556515e-01 6.57301180504999988524206244733250e-03
            9.27528685716000000915926193556515e-01 1.51382269814000006147303167836071e-02 5.73330873025999984693434896598774e-02 6.57301180504999988524206244733250e-03
            1.51382269814000006147303167836071e-02 9.27528685716000000915926193556515e-01 5.73330873026000054082373935671058e-02 6.57301180504999988524206244733250e-03
            8.15962504071100047653430920036044e-01 1.65971996956500000841572273202473e-01 1.80654989723999515049968067614827e-02 1.21440963474499993846045242662512e-02
            8.15962504071100047653430920036044e-01 1.80654989724000000772541341120814e-02 1.65971996956499945330421041944646e-01 1.21440963474499993846045242662512e-02
            1.65971996956500000841572273202473e-01 8.15962504071100047653430920036044e-01 1.80654989723999515049968067614827e-02 1.21440963474499993846045242662512e-02
            1.80654989724000000772541341120814e-02 8.15962504071100047653430920036044e-01 1.65971996956500000841572273202473e-01 1.21440963474499993846045242662512e-02
            1.65971996956500000841572273202473e-01 1.80654989724000000772541341120814e-02 8.15962504071100047653430920036044e-01 1.21440963474499993846045242662512e-02
            1.80654989724000000772541341120814e-02 1.65971996956500000841572273202473e-01 8.15962504071100047653430920036044e-01 1.21440963474499993846045242662512e-02
            3.16547555637800015482241633435478e-01 1.86886898773000013906919747341817e-02 6.64763754484899993535407247691182e-01 1.58399933165999998174733320865926e-02
            6.64763754484899993535407247691182e-01 1.86886898773000013906919747341817e-02 3.16547555637800015482241633435478e-01 1.58399933165999998174733320865926e-02
            1.86886898773000013906919747341817e-02 6.64763754484899993535407247691182e-01 3.16547555637800015482241633435478e-01 1.58399933165999998174733320865926e-02
            1.86886898773000013906919747341817e-02 3.16547555637800015482241633435478e-01 6.64763754484899993535407247691182e-01 1.58399933165999998174733320865926e-02
            3.16547555637800015482241633435478e-01 6.64763754484899993535407247691182e-01 1.86886898772999909823511188733391e-02 1.58399933165999998174733320865926e-02
            6.64763754484899993535407247691182e-01 3.16547555637800015482241633435478e-01 1.86886898772999909823511188733391e-02 1.58399933165999998174733320865926e-02
            1.92662192491999985366479819504093e-02 4.90366890375399988588611677187146e-01 4.90366890375399988588611677187146e-01 1.74658973517999999147676248867356e-02
            4.90366890375399988588611677187146e-01 1.92662192491999985366479819504093e-02 4.90366890375400044099762908444973e-01 1.74658973517999999147676248867356e-02
            4.90366890375399988588611677187146e-01 4.90366890375399988588611677187146e-01 1.92662192492000783339278768835356e-02 1.74658973517999999147676248867356e-02
            8.75134669581000002036574869634933e-02 8.24973066083700024009317530726548e-01 8.75134669581999480314493666810449e-02 1.91832266972499984603039990815887e-02
            8.75134669581000002036574869634933e-02 8.75134669581000002036574869634933e-02 8.24973066083799944081533794815186e-01 1.91832266972499984603039990815887e-02
            8.24973066083700024009317530726548e-01 8.75134669581000002036574869634933e-02 8.75134669581999757870249823099584e-02 1.91832266972499984603039990815887e-02
            9.35526036219000017846170180746412e-02 2.07986542316699996035822550766170e-01 6.98460854061400016057348238973645e-01 2.89184745605000001478224191941990e-02
            9.35526036219000017846170180746412e-02 6.98460854061299984962829512369353e-01 2.07986542316800027130341277370462e-01 2.89184745605000001478224191941990e-02
            2.07986542316699996035822550766170e-01 9.35526036219000017846170180746412e-02 6.98460854061400016057348238973645e-01 2.89184745605000001478224191941990e-02
            6.98460854061299984962829512369353e-01 9.35526036219000017846170180746412e-02 2.07986542316800027130341277370462e-01 2.89184745605000001478224191941990e-02
            6.98460854061299984962829512369353e-01 2.07986542316699996035822550766170e-01 9.35526036220000190013479368644767e-02 2.89184745605000001478224191941990e-02
            2.07986542316699996035822550766170e-01 6.98460854061299984962829512369353e-01 9.35526036220000190013479368644767e-02 2.89184745605000001478224191941990e-02
            9.74892983467000040498362523067044e-02 5.38008859514899961951073237287346e-01 3.64501842138400089510241741663776e-01 3.62910843697000010110720324973954e-02
            3.64501842138300002904571783801657e-01 9.74892983467000040498362523067044e-02 5.38008859515000104067894426407292e-01 3.62910843697000010110720324973954e-02
            5.38008859514899961951073237287346e-01 9.74892983467000040498362523067044e-02 3.64501842138400033999090510405949e-01 3.62910843697000010110720324973954e-02
            5.38008859514899961951073237287346e-01 3.64501842138300002904571783801657e-01 9.74892983468000351443549789109966e-02 3.62910843697000010110720324973954e-02
            3.64501842138300002904571783801657e-01 5.38008859514899961951073237287346e-01 9.74892983468000906555062101688236e-02 3.62910843697000010110720324973954e-02
            9.74892983467000040498362523067044e-02 3.64501842138300002904571783801657e-01 5.38008859515000104067894426407292e-01 3.62910843697000010110720324973954e-02
            2.21714589487299995607116898099775e-01 5.56570821025299977691247477196157e-01 2.21714589487399971190484393446241e-01 4.48928262053499976813597527325328e-02
            5.56570821025299977691247477196157e-01 2.21714589487299995607116898099775e-01 2.21714589487400026701635624704068e-01 4.48928262053499976813597527325328e-02
            2.21714589487299995607116898099775e-01 2.21714589487299995607116898099775e-01 5.56570821025399897763463741284795e-01 4.48928262053499976813597527325328e-02
            3.86047166929600005236267179498100e-01 2.27905666140799989527465641003801e-01 3.86047166929600060747418410755927e-01 5.17272266808500025270234345953213e-02
            2.27905666140799989527465641003801e-01 3.86047166929600005236267179498100e-01 3.86047166929600005236267179498100e-01 5.17272266808500025270234345953213e-02
            3.86047166929600005236267179498100e-01 3.86047166929600005236267179498100e-01 2.27905666140800045038616872261628e-01 5.17272266808500025270234345953213e-02
            ];
        
    case  16
        
        % ALG. DEG.:   16
        % PTS CARD.:   55
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 7.8e-13
        
        xyw_bar=[
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 3.10129992550000005771537736620758e-04
            0.00000000000000000000000000000000e+00 1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 3.15758735599999988078595558604889e-04
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 1.00000000000000000000000000000000e+00 3.54330077950000001639485613935676e-04
            9.39886358357699958077091650920920e-01 4.98487446340000004557646562375339e-03 5.51287671789000427446936214437301e-02 2.75818580839999995604761018341833e-03
            5.43806683058000034369783293186629e-02 9.38640561861699973533745833265129e-01 6.97876983250000915148802960175090e-03 3.13462038279999988601987048753017e-03
            9.39400491640000022408596436207517e-03 5.26424462697000009048231561337161e-02 9.37963548813899983258579595712945e-01 3.92657044129999990150770017294235e-03
            1.64345086361999993818105991749690e-02 9.46903551735099990871447062090738e-01 3.66619396287000132161892906879075e-02 4.72757419320000028123418545078493e-03
            9.46948726986200028932216810062528e-01 3.63373677167000003196406510141969e-02 1.67139052970999707481425389232754e-02 4.89122556354999998146082518246658e-03
            4.26604005767999977516424792156613e-02 1.51224541798999998626351626285214e-02 9.42217145243300024937127545854310e-01 4.99308217444999983408715493737873e-03
            1.22269495439000005360119516240047e-02 8.69377351066400017032265168381855e-01 1.18395699389699959880317692295648e-01 6.87769094080000007590536270640769e-03
            8.67369652104699961903122584772063e-01 1.20491728577399995203656146713911e-01 1.21386193179000428932212685140257e-02 7.04895890199999999858881949421630e-03
            8.45674402138899994341159072064329e-01 1.57763967870000014481757233397730e-02 1.38549201074100014619006060456741e-01 7.48234321684999956897366146790773e-03
            1.39575963210299996974939062965859e-01 8.44812087037499992447919794358313e-01 1.56119497522000383327167583047412e-02 7.80487518060000041303370110767901e-03
            1.31782174323099993129559948101814e-01 1.35009605583999992006738466443494e-02 8.54716865118499979914190589624923e-01 7.88418466740000052628278837119069e-03
            1.57955126300000012196100840355939e-02 1.45527493853600004003467915936199e-01 8.38676993516400015593603711749893e-01 8.78972731915000048130703902415917e-03
            7.36546288443600039030911830195691e-01 1.55697540907999993492483170598462e-02 2.47883957465599963354563328721269e-01 1.02056920135000003119207789836764e-02
            1.39688430330000001866608272393933e-02 7.37983689445000035078692235401832e-01 2.48047467521999909223495706100948e-01 1.04781439308000001026099923251422e-02
            2.54789518603899978188565000891685e-01 7.29761568977099983612788491882384e-01 1.54489124190000381986465072259307e-02 1.05356706499000005866628626449710e-02
            7.31638652255499954968342990468955e-01 2.54307668331499980851617692678701e-01 1.40536794130000641800393168523442e-02 1.08823380101000007774691979989257e-02
            1.57253728950999989433245929149052e-02 2.69623979579100003789449147006962e-01 7.14650647525799986858885404217290e-01 1.11144204349499994832761728957848e-02
            2.66230284364699976151769078569487e-01 1.44783956308000005819280886498746e-02 7.19291320004500023266302832780639e-01 1.12093346841000002234167709502799e-02
            8.67350406521400052461956420302158e-01 5.91679410399999980429974755224976e-02 7.34816524385999425561522002681158e-02 1.15061308496499994924455734235380e-02
            7.41493666957000058248539176020131e-02 8.63478257506099966533952283498365e-01 6.23723757981999860078303754562512e-02 1.18406951249999999736806088890262e-02
            1.59285948359999991741364766539846e-02 4.19123895523800027440586291049840e-01 5.64947509640200018488087607693160e-01 1.28732321684000006190018439156120e-02
            1.56061028068000002405613102496318e-02 5.80922292114600002932434108515736e-01 4.03471605078600026317303672840353e-01 1.28978400803999998514015601358551e-02
            5.91009481748399956302364444127306e-01 1.59251452650999991700331293031923e-02 3.93065372986500016772026810940588e-01 1.29036163804999992610333947595791e-02
            4.03477149688900027513227541930974e-01 5.80670036810399969873230929806596e-01 1.58528135007000026135415282624308e-02 1.30171616029500000072394172434542e-02
            5.69474562852600008078240989561891e-01 4.14949514630200022580197583010886e-01 1.55759225171999693415614274272230e-02 1.32884070804499994367109394488580e-02
            6.78493700649999975649251382492366e-02 7.61218678590999953126328136931988e-02 8.56028762075899951611290816799738e-01 1.32892380915500003119023020303757e-02
            4.26596859027200014935488070477732e-01 1.57509692312000001979122032480518e-02 5.57652171741599977927705822366988e-01 1.33766164619000000463033828168591e-02
            6.70982507890000023920151761558373e-02 7.74189831242099946173595981235849e-01 1.58711917968900051434388842608314e-01 1.87893903320500016418392874584242e-02
            7.52831023147999989220124916755594e-01 8.19119495638999939313151799069601e-02 1.65257027288100016848559903337446e-01 1.91532947097499996191860560657005e-02
            7.75372778355699954211388558178442e-01 1.57712845729199996336333811086661e-01 6.69143759151000494522776307348977e-02 1.92424847512500003798852077352421e-02
            1.68907315778699995689748902805150e-01 7.50394309974199980040054924756987e-01 8.06983742471000242701961724378634e-02 1.94809912925999993704717638820512e-02
            1.68733583291899996714491294369509e-01 7.08311507268000045689504418078286e-02 7.60435265981299957083194840379292e-01 1.97302055773499983304386518057072e-02
            8.21244708436000003803911795330350e-02 1.76299662677100010821362729984685e-01 7.41575866479299961042670474853367e-01 2.06182389048999992198218933481257e-02
            6.28870536334499963260213917237706e-01 8.07744953317000025094785087276250e-02 2.90354968333800034230307574034669e-01 2.56436219241500014398571494211865e-02
            8.11413015265999987279954552832351e-02 3.05437358977599993092866270671948e-01 6.13421339495799911034623619343620e-01 2.58202820967500003279671005884666e-02
            2.96911206508000014014925227456843e-01 6.22748598887099991117111130733974e-01 8.03401946049000503791148730670102e-02 2.59115021134499988464394704124061e-02
            7.67542314170999995237565372008248e-02 6.24724714954600024441333516733721e-01 2.98521053628299948279334330436541e-01 2.64263994090499984568332081380504e-02
            6.22302233384499969837122534954688e-01 3.01148582116600005420536945166532e-01 7.65491844989000247423405198787805e-02 2.69252786513499986487119031153270e-02
            3.10378628805099976428749641854665e-01 7.79098365078999960386951784130360e-02 6.11711534687000013654767371917842e-01 2.70947664659500001005376645935030e-02
            8.19218215187000053933985554976971e-02 4.60363303835099990646284595641191e-01 4.57714874646200031715892464490025e-01 2.92368573222000008304810592107970e-02
            4.71702266501299982692785306426231e-01 8.21554006796999941597547945093538e-02 4.46142332819000009269672091249959e-01 2.96431584181500000496622959644810e-02
            4.54660341525000011309032288409071e-01 4.63756503388999985482143983972492e-01 8.15831550860000032088237276184373e-02 2.97179138374500002861555714162023e-02
            1.70109133923700001078671562027012e-01 6.42227780818800053630468482879223e-01 1.87663085257499973046435570722679e-01 3.15900127931499977163021242176910e-02
            6.40600432948699949342596937640337e-01 1.89829353725600002977813574034371e-01 1.69570213325700047679589488325291e-01 3.16463422576499994742782462253672e-02
            1.91226758371700000793680374044925e-01 1.73995568534300010243853762403887e-01 6.34777673094000016718041479180101e-01 3.20353680885999969607524917591945e-02
            1.88531576706999992598667859056150e-01 4.79891407040600015410092282763799e-01 3.31577016252400047502391089437879e-01 4.06020297959000034393461930903868e-02
            4.77292995769100003577989355108002e-01 3.34835659811899999382944770331960e-01 1.87871344418999997039065874560038e-01 4.07218756764999995079179484491760e-02
            3.12697462175999973954532151765306e-01 4.95797219725900017994746349359048e-01 1.91505318098100063561872730133473e-01 4.07339600620499966976417738351302e-02
            4.96122594594600008210960595533834e-01 1.92755366890399987278570392845722e-01 3.11122038515000087777195858507184e-01 4.07525274041999965857208110264764e-02
            1.92880531286699991033728451839124e-01 3.16101580726100006568657363459351e-01 4.91017887987199974642038569072611e-01 4.07582332469499999949569257751136e-02
            3.36004145381600005659805674440577e-01 1.89489280128999987296012363913178e-01 4.74506574489399923777455114759505e-01 4.08465529811500027701498538590386e-02
            3.33728055084800023788460521245725e-01 3.34357102181099985482859437979641e-01 3.31914842734099935217528809516807e-01 4.61609167265499970023157061405072e-02
            ];
        
    case  18
        
        % ALG. DEG.:   18
        % PTS CARD.:   66
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.8e-16
        
        xyw_bar=[
            1.16731059668422967878775864392082e-02 9.81256595128901731861503776599420e-01 7.07029890425592277836130961077288e-03 1.25828784932243364046156042235225e-03
            9.81003085838796140549789015494753e-01 7.14625048631965624490458566242523e-03 1.18506636748842032053063988428221e-02 1.26367260036095037899206428022580e-03
            1.06966317091656323151616092559379e-02 1.15153933375962483354193466311699e-02 9.77787974953238103736907760321628e-01 1.66346476665857529765590872017356e-03
            9.38247698355051062968357200588798e-01 4.95570591340603744390236329309118e-02 1.21952425108885625926191664802900e-02 4.07517460627058666911004536359542e-03
            1.26627518417216609891706724511096e-02 9.37012362061508063959536229958758e-01 5.03248860967703270929973768943455e-02 4.30677628708067260915326457393348e-03
            5.98109409983654249920448364719050e-02 1.21364578921858024679325538386365e-02 9.28052601109448871419260740367463e-01 4.38933730896492405731956765180257e-03
            1.37363297926714009872251054389380e-02 6.12783625596970207838332100891421e-02 9.24985307647631649352604199521011e-01 4.85497927808363743257258349217409e-03
            9.22952795940536918628538387565641e-01 1.41128270602437974612364257609443e-02 6.29343769992192769713312827661866e-02 5.12331059574436427361820989290209e-03
            6.33107354992725884779147804692911e-02 9.22019729172741109302080531051615e-01 1.46695353279863160977924962935504e-02 5.41988441703700543627952868064312e-03
            1.17265100334620151617537331389940e-02 1.50052047522906850662849365107832e-01 8.38221442443631170604589897266123e-01 6.46926950879339861949501866433820e-03
            1.55472058732349788234827769883850e-01 8.32514712158920699991426772612613e-01 1.20132291087294840181698418746237e-02 6.81699117914837462994803729543491e-03
            8.34329388898202961577510450297268e-01 1.25228158758818042262106473572203e-02 1.53147795225915234196278902345512e-01 6.92386640733264058233942250808468e-03
            8.50163803195680856106264400295913e-01 1.37199750873573111942249624917167e-01 1.26364459307460319514859747869195e-02 6.97107700524267348551887835128582e-03
            1.28816350521985811816705336241284e-02 8.47762706347928829408999717998086e-01 1.39355658599872644920480979635613e-01 7.20606999838006162528714071413560e-03
            1.51080160895852660951632628894004e-01 1.36526924039371260616304581958502e-02 8.35267146700210294518740283820080e-01 7.68517277670055943411542287435623e-03
            1.01917879216608664449950083508156e-02 5.77043861834499960394850859302096e-01 4.12764350243839173160154132347088e-01 8.12449011263043953390283746784917e-03
            2.81337239930328442127915877790656e-01 7.06685375962319062281835613248404e-01 1.19773841073524955902485089609399e-02 8.48591521400751148951346891635694e-03
            7.12437462850067815267607329587918e-01 1.24569780989904812379398535426844e-02 2.75105559050941694820835436985362e-01 8.50442662106710468850057083045613e-03
            2.76302525086301431489488322768011e-01 1.21741311385635056496745676213322e-02 7.11523343775135153066457860404626e-01 8.54767603373146091227940956969178e-03
            1.09658368560642989486542120403101e-02 4.19430671246641517679876187685295e-01 5.69603491897294089696401897526812e-01 8.69444272795687027322042439436700e-03
            4.28911051788389785510702267856686e-01 5.59961606746893436614698202902218e-01 1.11273414647167223634482979832683e-02 8.72719812193595116667221844863889e-03
            4.21542055511432756098599838878727e-01 1.16475994784655950686769187996106e-02 5.66810345010101634954935434507206e-01 8.92033786433164312124066697151648e-03
            5.71125859044400052688672531076008e-01 1.18218313988502476269415453202782e-02 4.17052309556749711827450255441363e-01 8.92234319396820879655152225495840e-03
            5.82686827051093314544516488240333e-01 4.05788958117719755769314815552207e-01 1.15242148311869296861686962074600e-02 8.95231687761671030001409121723555e-03
            1.30567806713243420718706389038744e-02 2.72502375086800474601744781466550e-01 7.14440844241875261388940998585895e-01 9.06298781003535107447799390456566e-03
            1.30760400963916122857311208349529e-02 7.22471252323362400638018243625993e-01 2.64452707580245971463739351747790e-01 9.23924194410074249572684124132138e-03
            7.26343706240680786478947084106039e-01 2.60298401925061928530169552686857e-01 1.33578918342572849908833632071037e-02 9.28967821855570564937298883023686e-03
            6.87230068637360302918892784873606e-02 6.31417277209678357019484451484459e-02 8.68135265415296175639525699807564e-01 1.01608575888281916360655898756704e-02
            8.65230210152928114197834474907722e-01 7.20611837337742783260452483773406e-02 6.27086061132976074761202767149371e-02 1.06885830904604756802500631351904e-02
            6.48599071037404689699101822952798e-02 8.59043354390996882763431585772196e-01 7.60967385052626621444460397469811e-02 1.15958427049141603315529636120118e-02
            1.48349494336221521262331179968896e-01 7.88878835223944308019383697683224e-01 6.27716704398341152071338910900522e-02 1.37213355429586653100315629671968e-02
            6.24359898395992346520344540294900e-02 1.49393549935431940722807553356688e-01 7.88170460224968838502945800428279e-01 1.45150961170182456888566591146628e-02
            7.87136901173496439376719990832498e-01 6.56382042756523814697189322941995e-02 1.47224894550851193031348884687759e-01 1.47261369252703576415708397462367e-02
            5.19104921609673947147101102927991e-02 5.25563569560499987964874435419915e-01 4.22525938278532575687052030843915e-01 1.49718125814584726573919226666476e-02
            1.54312992744386756616137290620827e-01 7.16383926916974306253749205097847e-02 7.74048614563915826636275596683845e-01 1.53513474059392852205663260178881e-02
            2.61784274560300012080205078746076e-01 6.21479485288090535699545569059410e-02 6.76067776910890927410946460440755e-01 1.62631682931342509690342268413588e-02
            7.66725787281274606677072824822972e-01 1.65821155483138971131751304710633e-01 6.74530572355864221911758704663953e-02 1.63942104253045027595359073302461e-02
            2.58210367662758732976158171368297e-01 6.80011976613870805508099692815449e-01 6.17776557233704615157421358162537e-02 1.65617337596012416389701371599585e-02
            6.79065925147462334798831307125511e-02 7.57151543778185054378582208300941e-01 1.74941863707068767652685892244335e-01 1.73083763437293959575047352927868e-02
            5.29357827480446907486566487932578e-01 4.12150384110698542183826020846027e-01 5.84917884088545503296074912213953e-02 1.73540686988041432947671438569159e-02
            6.66036150484166367125027363726986e-02 2.61251308788644831526681855393690e-01 6.72145076162938615027542255120352e-01 1.73686024701916047474625770519197e-02
            5.85675461899543767230191804173955e-02 3.90223611453477114974219830401125e-01 5.51208842356568418097140238387510e-01 1.74264381227117563788464593699246e-02
            6.44535360410797841801411323103821e-02 6.37362655976104197641518567252206e-01 2.98183807982815962667189069179585e-01 1.74300780592989701389683432353195e-02
            6.74813842915122696908269972482231e-01 6.37583342061289748414765199413523e-02 2.61427822878748328250253507576417e-01 1.77735784987432983428323751695643e-02
            3.91460231036899974910170385555830e-01 5.50323809056282087226463772822171e-01 5.82159599068179378633658416219987e-02 1.80091498191344220880516502347746e-02
            6.48770149230723958133637552236905e-01 2.83672836026290065003507834262564e-01 6.75570147429859768628546135005308e-02 1.81463142921354687875368227878425e-02
            3.94649822040804787448564638907555e-01 6.05175522553690475868748421817145e-02 5.44832625703826178842348326725187e-01 1.90948851041583926579914276544514e-02
            5.39013715193328413199935766897397e-01 6.11990176936391058748476723394560e-02 3.99787267113032473986322656855918e-01 1.96126400058931338321599469054490e-02
            1.62789508278483124881219623603101e-01 6.86132214103464810328603107336676e-01 1.51078277618052037034601653431309e-01 2.41355062943759378357722056307466e-02
            6.81243632264075471560715868690750e-01 1.56796834589900596634848284338659e-01 1.61959533146023931804435846970591e-01 2.44956060783115894818884328287822e-02
            1.54283287802026880530803509827820e-01 1.66751262401978767035970463439298e-01 6.78965449795994380188801642361796e-01 2.48610416936092606943198290991859e-02
            2.52272775044500019436810589468223e-01 2.50480393339460316592948174729827e-01 4.97246831616039719481392467059777e-01 2.53532868492960770934185887881540e-02
            2.54798153240699987165385209664237e-01 4.99409064904299981879631786796381e-01 2.45792781855000030954983003539382e-01 2.54885997021513967253358856623890e-02
            1.48558054919430054807705232633452e-01 5.75602309608733597734442355431383e-01 2.75839635471836319702276796306251e-01 2.60680031833592229517648775072303e-02
            2.93023960643626679711815086193383e-01 5.65689735416167849280100199393928e-01 1.41286303940205471008084714412689e-01 2.61730437462385813474075035856004e-02
            2.80899127230993306270079301611986e-01 1.43792157424766253726033937709872e-01 5.75308715344240440003886760678142e-01 2.62220341775821381979927338079506e-02
            4.82098959297124984857418894534931e-01 2.51855753586498587459630016383016e-01 2.66045287116376427682951089082053e-01 2.63729822411241827950778571221235e-02
            5.64187824544361338219289336848306e-01 1.46296674315256225451520322167198e-01 2.89515501140382436329190340984496e-01 2.64724531863811049503532046855980e-02
            1.30769964434413243914434588077711e-01 4.48957758611656376768195286786067e-01 4.20272276953930379317370125136222e-01 2.71197797250435669591528409227976e-02
            1.47969222194773153411517796484986e-01 3.00117438682896364721131021724432e-01 5.51913339122330537378502413048409e-01 2.71735101709683242454662632781037e-02
            5.63868422294613202971902410354232e-01 2.81377208929745936671906747506000e-01 1.54754368775640860356190842139767e-01 2.73550274319399565337018742638975e-02
            4.36115742879063184034293954027817e-01 4.25205344642040183877895742625697e-01 1.38678912478896632087810303346487e-01 2.78644172956311816924390711847082e-02
            3.60326393528606292449012471479364e-01 2.59919000488856355435274281262537e-01 3.79754605982537352115713247258100e-01 2.88867132116526759144647940047435e-02
            4.22418833467427257133408602385316e-01 1.45323844330263851620799187003286e-01 4.32257322202308891245792210611398e-01 2.92696890811342397276106908066140e-02
            3.71900183305260034760664211717085e-01 3.78012270356698043549670273932861e-01 2.50087546338041977200816745607881e-01 3.04519625339803902408508662347231e-02
            2.41364500692875016962801737463451e-01 3.84756328493939148138736072723987e-01 3.73879170813185779387310958554735e-01 3.18636982224730636037435260732309e-02
            ];
        
    case  20
        
        % ALG. DEG.:   20
        % PTS CARD.:   78
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.0e-16
        
        xyw_bar=[
            8.94113371120544217496295402725082e-03 8.69832937019905140518805097826771e-03 9.82360536918595483868443807295989e-01 1.08722726992999639550330215342910e-03
            9.79262262980672426593287127616350e-01 1.02644133743667476804661831124577e-02 1.04733236449608257262466892711927e-02 1.44935676325574219758252070278104e-03
            1.05475382111858816819305317835642e-02 9.78551420251513315307079210469965e-01 1.09010415373007596429033583262935e-02 1.54230146682676348864060589249902e-03
            2.37770619471529233088236132687143e-03 6.36551098603544696086231624576612e-02 9.33967183944930168237874568148982e-01 1.72008165518022326616331962867434e-03
            6.30425115794740753560887469575391e-02 4.15063475086186860901760198316879e-03 9.32806853669664093331448384560645e-01 2.09492360058796981900353095795708e-03
            9.30842249673007549048975306504872e-01 4.80534822625924146199949404945073e-03 6.43524021007332025501312955384492e-02 2.23690257489724594192059825559227e-03
            6.29076555490187122599010649537377e-02 9.31679006948127330645093024941161e-01 5.41333750285399872836933354847133e-03 2.35272104071028876487448044940720e-03
            9.31596224638061132239386097353417e-01 6.26264881801440742581377207898186e-02 5.77728718179479350247618185676401e-03 2.44339678751489663283136266613838e-03
            6.19516894144997096183224982723914e-03 9.29358705856405231671146793814842e-01 6.44461252021447661419983887753915e-02 2.59638216844565060045790794163167e-03
            2.87125819236766499642499184119515e-02 3.10202122997135891491193859792475e-02 9.40267205776609715783820320211817e-01 3.70365294904031333578231510728074e-03
            9.29384447830515436628218139958335e-01 3.42152968218708067316136123281467e-02 3.64002553476137566401682477135182e-02 3.98777051504645759644196445492526e-03
            3.75457566621083954627735579379078e-02 9.25786888466940571795760206441628e-01 3.66673548709510743748296590638347e-02 4.17752614547965608199353582108415e-03
            8.68957390638382641545600648669279e-03 1.58497125150980244789877815492218e-01 8.32813300942635947876624413765967e-01 4.80833304318707601548998908924659e-03
            1.54759705396456032078233988613647e-01 8.36360665768839717237881359324092e-01 8.87962883470427843946026769117452e-03 4.81591289252253969971340197275822e-03
            8.33102529418505488578716722258832e-01 8.92572448244856840593453028986914e-03 1.57971746099045934341731367567263e-01 4.92887303791673860725586209241555e-03
            8.37423107352594398022915811452549e-01 1.52916730407855333551125909252733e-01 9.66016223955026842595827929471852e-03 5.13289401506832922150724840548719e-03
            1.55936250523392189171190125307476e-01 9.49662400580270418815054256356234e-03 8.34567125470805026843379437195836e-01 5.15940515555492890120703464162943e-03
            9.85996420952243816937521359022867e-03 8.34221149359543723278420657152310e-01 1.55918886430933878450844076724024e-01 5.31455008152321908138082662276247e-03
            4.05587373328940736794123722575023e-01 7.43893020079155944251914078790833e-03 5.86973696470267647384844167390838e-01 5.34406534476825588464432570390272e-03
            5.96472789861799967958688739599893e-01 3.95633080931094838206263375468552e-01 7.89412920710519383504788493155502e-03 5.34845105051587419581204940755015e-03
            8.07478004157673352747526251960153e-03 4.03131942590241654222893430414842e-01 5.88793277368181655617718206485733e-01 5.45132308568089217482022235117256e-03
            7.50739777207003189385403629785287e-03 5.85160959468056574017680304677924e-01 4.07331642759873435721829082467593e-01 5.49498917872474033907703727663829e-03
            3.93676451923723857984072083127103e-01 5.97489659289885510773387977678794e-01 8.83388878639057573138870793627575e-03 5.67115276145129573043268678134154e-03
            5.84653072621227565264234726782888e-01 8.72504649681960300910965599996416e-03 4.06621880881952835196102569170762e-01 6.02678214649227540072873310350587e-03
            4.87080411211974706464644668812980e-01 2.02129229911995403345947863726906e-02 4.92706665796825815650805679979385e-01 6.98095969102919232385229264536974e-03
            2.68351281178442424568686419661390e-01 7.20234008866831643125294704077533e-01 1.14147099547259323060188762610778e-02 7.05739957682016000317659631946299e-03
            7.22395628874778816808088777179364e-01 2.66239936645638941570268798386678e-01 1.13644344795822416216424244339578e-02 7.09651735229200903570800917918859e-03
            2.71682674235739995971528060181299e-01 1.12882698808235912291131342044537e-02 7.17029055883436461371616132964846e-01 7.21063381341643765132243260040923e-03
            1.12580842045890989339129717450305e-02 7.16969596332504899294235656270757e-01 2.71772319462906031262150463589933e-01 7.23521734274507749562488356787071e-03
            1.15034734369742191323338076358596e-02 2.74006711016551274262553761218442e-01 7.14489815546474504870388955168892e-01 7.24748849360278609571528463106915e-03
            7.14052590056392100237303566245828e-01 1.13511560497063217906044130245391e-02 2.74596253893901576237368544752826e-01 7.26933878470379605390983002166649e-03
            4.90287105311137949570365890394896e-01 4.93649184146848774012994454096770e-01 1.60637105420132764166396555083338e-02 7.29820954627875655151836298273338e-03
            2.01423425209245159084314735764565e-02 4.83257345960130979545255058837938e-01 4.96600311518944514954654323446448e-01 7.36572892331423690476244203750866e-03
            3.61107464858592894230149283885112e-02 9.35679501581969663703830519807525e-02 8.70321303355943709512132500094594e-01 8.37319816520741555532225675051450e-03
            8.60799881985092274483406526996987e-01 3.97379067075453867863288337503036e-02 9.94622113073623387302646392527095e-02 8.44777502287684389781219351789332e-03
            1.00589152600109088542268409582903e-01 8.58634341935195832817839800554793e-01 4.07765054646951341510430211201310e-02 8.47113314420841305441012281107760e-03
            9.18740717058512956771920698884060e-02 3.95513001973352915086046266424091e-02 8.68574628096813405875309399561957e-01 8.65350860477442918039514552219771e-03
            8.60488829619093453793254866468487e-01 9.66224057079418385818314618518343e-02 4.28887646729647076249136716796784e-02 8.72622732463380207379444186699402e-03
            4.39842178673168632263568156304245e-02 8.56188634910677426326230943232076e-01 9.98271472220056965696244333230425e-02 8.86086110796604001227372293669760e-03
            2.01101760673522977906912956314045e-01 7.44911583562643420108884129149374e-01 5.39866557638335464730516832787544e-02 1.41412012011636629288391375780520e-02
            7.44999372626340083947127368446672e-01 5.36865638165824748950782918655023e-02 2.01314063557077427280006531873369e-01 1.42498356244064386388892273771489e-02
            5.32186641309854713277616156119620e-02 1.96375427593497742240202796892845e-01 7.50405908275516786432035587495193e-01 1.42502823269358858820909929931986e-02
            7.45398464740034905062771031225566e-01 1.98206580555018330169758655756596e-01 5.63949547049467647674703130178386e-02 1.50323611738851174030040525053664e-02
            1.95728993287614144724884113202279e-01 5.55713833156083122921664596560731e-02 7.48699623396777536044055523234420e-01 1.51015638541140365808113088519349e-02
            1.09253205798743646681714380974881e-01 6.10003618241300005209382106841076e-01 2.80743175959956348108903512184042e-01 1.51993568038457874808111469633332e-02
            5.67625702000490173215752065516426e-02 7.40912189495913087178280420630472e-01 2.02325240304037867744568757188972e-01 1.52834398036998126563235089747650e-02
            4.83837933474768472352778303502419e-02 6.07513566097787838238275526236976e-01 3.44102640554735272893083219969412e-01 1.53033706501137269084811265429380e-02
            1.08061280976018470045829644732294e-01 1.12208151043696469084665068294271e-01 7.79730567980285060869505286973435e-01 1.54665034100298377089943713258435e-02
            6.18560590099037632683121046284214e-01 2.69875370303500006219366014192929e-01 1.11564039597462361097512939522858e-01 1.54886910417429829456281709099130e-02
            7.72129601349661243148148059844971e-01 1.11411739533305548244257465739793e-01 1.16458659117033208607594474415237e-01 1.56573125272326148238466458906260e-02
            6.11573480113248968415007311705267e-01 3.38936767793083126854014608397847e-01 4.94897520936679047309780798968859e-02 1.56786746695895401471076979760255e-02
            3.38132610337598005223469499469502e-01 4.94693938787471032547671256907051e-02 6.12397995783654947032914606097620e-01 1.57160234643337025361997660866109e-02
            1.17308412825418845182312566066685e-01 7.69645130979520919467518069723155e-01 1.13046456195060263105744979839073e-01 1.57591071946765054478944989568845e-02
            2.67455126059625236667471881446545e-01 1.11571880815410667819875811801467e-01 6.20972993124964123268227922380902e-01 1.62124068992756907581220815472989e-02
            6.54210016002574556992499310581479e-01 1.90654831469986796310678300869768e-01 1.55135152527438646696822388548753e-01 1.73756076192967609528317041167611e-02
            5.38297481157758231362642220574344e-02 3.35861682684902740358978689982905e-01 6.10308569199321526710377838753629e-01 1.75196727463413637337907857727259e-02
            1.84884032411679721796815556444926e-01 1.55183152385134082162920776681858e-01 6.59932815203186251551414898131043e-01 1.75358710155161037769211418435589e-02
            3.37626710474410451912063990675961e-01 6.08140259629467938573554874892579e-01 5.42330298961216650255323656892870e-02 1.76064607667074622676661732612047e-02
            6.06710203449961738009221789980074e-01 5.42632795598224151878419263539399e-02 3.39026516990215853741830187573214e-01 1.76307752490747829399175827802537e-02
            4.61261408549577645032968575833365e-01 6.88176670721690475485132765243179e-02 4.69920924378253279662942532013403e-01 1.83201610171350656730560046980827e-02
            1.52546536567121610028507916467788e-01 6.51024084574876371611651393322973e-01 1.96429378858002046115416305838153e-01 1.83866553835163315522294169568340e-02
            7.00582543543042252087715837660653e-02 4.66190439274151935400425372790778e-01 4.63751306371543825513015235628700e-01 1.85837831468433078985214024214656e-02
            4.70420137903155399783372558886185e-01 4.63482645535340076481389814944123e-01 6.60972165615045237352376261696918e-02 1.86685785803141872374322218774978e-02
            1.21646169374596019308043537421327e-01 2.38149487551563759923922702910204e-01 6.40204343073840220768033759668469e-01 2.01986673293933008399569928315032e-02
            6.37140405270220533395786333130673e-01 1.23839938451338588132344398218265e-01 2.39019656278440878471869268651062e-01 2.06790020318823875022928859834792e-02
            2.37990451511855244381976604017837e-01 6.37021645232653477286532961443299e-01 1.24987903255491250575914818909951e-01 2.10978895935022142671400047220231e-02
            1.48392985717712366788845201881486e-01 4.89418857778011140435836523465696e-01 3.62188156504276437264167043394991e-01 2.47725502018576806961824843256181e-02
            3.59806957154975348700531867507380e-01 1.45288086625325219003457277722191e-01 4.94904956219699432296010854770429e-01 2.50209630570553537287636913788447e-02
            4.94144105509497388872830470063491e-01 3.61021638381830600739164083279320e-01 1.44834256108672010388005446657189e-01 2.52897293557296266863776423861054e-02
            1.44063068798082505317381674103672e-01 3.51350834188701666160881131872884e-01 5.04586097013215884032888425281271e-01 2.60018605093971615083159321102357e-02
            5.01976444000366672781865418073721e-01 1.43549166329309002554381891059165e-01 3.54474389670324296908177075238200e-01 2.60766783942758643977111177036932e-02
            3.55542383429807651307186233680113e-01 5.01649159950201317670348544197623e-01 1.42808456619990975511313990864437e-01 2.62449576178954320504477237818719e-02
            2.44343954077136132907455134954944e-01 2.40605212910407678661428576560866e-01 5.15050833012456132919965057226364e-01 2.99579881257902823354921650889082e-02
            2.43706498934185250559991686714056e-01 5.10901727705534658419139759644167e-01 2.45391773360280063265292938012863e-01 2.99804998713138690091817295524379e-02
            5.12220080732083804520016201422550e-01 2.45273797354294331718449484469602e-01 2.42506121913621863761534314107848e-01 2.99957636064386926055824744707934e-02
            2.52603831517775656578805865137838e-01 3.70031955509367738166304206970381e-01 3.77364212972856605254889927891782e-01 3.17066591724676702379603909776051e-02
            3.75989565285069105016191315371543e-01 2.50540661163056177507257871184265e-01 3.73469773551874717476550813444192e-01 3.17655930554187823644518573473761e-02
            3.72907798714407878737375767741469e-01 3.75375027754923540346254640098778e-01 2.51717173530668636427520823417581e-01 3.18603302836001148201638955015369e-02
            ];
        
    case  21
        
        % ALG. DEG.:   21
        % PTS CARD.:   91
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.0e-16
        
        xyw_bar=[
            3.55243919220995813673669161403268e-03 3.55243919219624861710604690756554e-03 9.92895121615593789776710309524788e-01 3.35221821934728742009568236426276e-04
            3.55243919220429599931110331567652e-03 9.92895121615643416745911054022145e-01 3.55243919215231240826824432588182e-03 3.35221821933162232500652555344800e-04
            9.92895121615600562137160522979684e-01 3.55243919219402426792897564666873e-03 3.55243919220541359491050137364709e-03 3.35221821934558901739248382156688e-04
            9.55354827372969173104877427249448e-01 8.78989290927685053278484161864981e-03 3.58552797177539780970612071087089e-02 2.27363040370187922609157560316362e-03
            3.58552797177463869471303326008638e-02 8.78989290927979956269400219071031e-03 9.55354827372973836041580852906918e-01 2.27363040370173827980915248758720e-03
            9.55354827372995485390561043459456e-01 3.58552797177243073867280998001661e-02 8.78989290928020722271085674037749e-03 2.27363040370189787436894235383988e-03
            8.78989290927446355328189753208790e-03 3.58552797177304483078330576972803e-02 9.55354827372995041301351193396840e-01 2.27363040370093597020151321430603e-03
            8.78989290927445314494104167124533e-03 9.55354827373050552452582451223861e-01 3.58552797176749926677530311280861e-02 2.27363040370103701784398886331928e-03
            3.58552797177701526587512148580572e-02 9.55354827372945747399057836446445e-01 8.78989290928411381997875650995411e-03 2.27363040370414520863207030743069e-03
            8.86526487904757964919610913057113e-01 1.08232974501700593417119478090171e-01 5.24053759354144166326960885271546e-03 2.60387926599255558821055522855659e-03
            8.86526487904712667820206206670264e-01 5.24053759349522689514611073491324e-03 1.08232974501792103549924206618016e-01 2.60387926597782605117603615951793e-03
            5.24053759350145108297791551876799e-03 1.08232974501748444029480822337064e-01 8.86526487904750193358438536961330e-01 2.60387926597915831880558634736644e-03
            5.24053759352011410549421199789322e-03 8.86526487904805704509669794788351e-01 1.08232974501674128475769975921139e-01 2.60387926598502992409089884517925e-03
            1.08232974501802803324324031564174e-01 8.86526487904688242913664453226374e-01 5.24053759350895376201151520945132e-03 2.60387926598233589853270686376163e-03
            1.08232974501755729868079924926860e-01 5.24053759349578460874363727839409e-03 8.86526487904748528023901599226519e-01 2.60387926597741535539309865043833e-03
            4.66397432149654941890837278606341e-02 9.06720513569923891594726228504442e-01 4.66397432151106627884473709855229e-02 3.27177164433340621554902050149849e-03
            4.66397432150000013084678585073561e-02 4.66397432150258764438355285619764e-02 9.06720513569974184697741748095723e-01 3.27177164433800887061165596492174e-03
            9.06720513570000052894215514243115e-01 4.66397432150000013084678585073561e-02 4.66397432149999457973166272495291e-02 3.27177164432620841416632373466200e-03
            2.07572045694564971674012099356332e-01 8.27592412838175708889032478055015e-03 7.84152030177053260828756720002275e-01 4.63689207664152742288665720593599e-03
            2.07572045694622564493414529351867e-01 7.84152030176993752874636811611708e-01 8.27592412838362712079742777859792e-03 4.63689207664253529722619973085784e-03
            7.84152030177069025995706397225149e-01 2.07572045694540824323226502201578e-01 8.27592412839014968106710057327291e-03 4.63689207664790339902260996041150e-03
            8.27592412838997620871950289256347e-03 7.84152030177113656961296328518074e-01 2.07572045694496387646665880311048e-01 4.63689207664724160201652480850498e-03
            8.27592412838307027456163922352061e-03 2.07572045694563389606202008508262e-01 7.84152030177053482873361645033583e-01 4.63689207664245983675499473974924e-03
            7.84152030176999970123574712488335e-01 8.27592412837836050032436219225929e-03 2.07572045694621676314994829226634e-01 4.63689207664035388245515889593662e-03
            8.58119489724999962021456667571329e-02 3.14836947700999972288293804467685e-02 8.82704356257399957996767625445500e-01 4.79688913117405028169182301667206e-03
            8.82704356257399957996767625445500e-01 3.14836947700999972288293804467685e-02 8.58119489725000517132968980149599e-02 4.79688913117855449119719679629270e-03
            3.14836947700999972288293804467685e-02 8.58119489724999962021456667571329e-02 8.82704356257399957996767625445500e-01 4.79688913117145340064828573645173e-03
            8.58119489724622208637327958058449e-02 8.82704356257399957996767625445500e-01 3.14836947701377933839239631197415e-02 4.79688913118266318375004786389582e-03
            8.82704356257399957996767625445500e-01 8.58119489723861289531825491394557e-02 3.14836947702139130500498254150443e-02 4.79688913117119752893557915740530e-03
            3.14836947701365027496578363752633e-02 8.82704356257399957996767625445500e-01 8.58119489724635808869379616226070e-02 4.79688913116526997881816640756369e-03
            6.68877823382516334405067937041167e-01 9.51507606247778067731779572113737e-03 3.21607100555005909203742930912995e-01 5.71239045834627447856268744885710e-03
            9.51507606247835140134139209067143e-03 3.21607100554938851733055571457953e-01 6.68877823382582725741940521402285e-01 5.71239045834591886025011220340275e-03
            9.51507606248120849090632589195593e-03 6.68877823382600045221124673844315e-01 3.21607100554918701185158624866745e-01 5.71239045835077608598284726326710e-03
            6.68877823382546310426732816267759e-01 3.21607100554971603312282013575896e-01 9.51507606248208626098517015634570e-03 5.71239045835149079205494970779000e-03
            3.21607100554981262252596252437797e-01 6.68877823382539427043980140297208e-01 9.51507606247936621457483852282166e-03 5.71239045834512782634506677936770e-03
            3.21607100554938130088089565106202e-01 9.51507606247852834313594172499506e-03 6.68877823382583391875755296496209e-01 5.71239045834613656804634729269310e-03
            4.37999954311300010445506813994143e-01 9.98597856805449532457963357501285e-03 5.52014067120645490760466600477230e-01 5.86084820867617106027180895466699e-03
            9.98597856805516492784136062255129e-03 5.52014067120596196858173243526835e-01 4.37999954311348638213985395850614e-01 5.86084820867968474267239997743673e-03
            4.37999954311300010445506813994143e-01 5.52014067120644713604349362867652e-01 9.98597856805527595014382313820533e-03 5.86084820867435740687767520284979e-03
            9.98597856805430623972075210303956e-03 4.37999954311300010445506813994143e-01 5.52014067120645712805071525508538e-01 5.86084820867653882164871603777101e-03
            5.52014067120551343847978387202602e-01 4.37999954311393047134970402112231e-01 9.98597856805560901705121068516746e-03 5.86084820867850686543221172541962e-03
            5.52014067120565998791903439268935e-01 9.98597856805341979602452795461431e-03 4.37999954311380557125943369101151e-01 5.86084820867607218103367827666261e-03
            7.97493107214751639766348034754628e-01 4.05093994118307862417616149741662e-02 1.61997493373417567052996446363977e-01 9.40985776160098603970727282330699e-03
            4.05093994118551278815765215313149e-02 1.61997493373374240599460449629987e-01 7.97493107214770624580069124931470e-01 9.40985776160365230968984917581110e-03
            4.05093994118959702110949194775458e-02 7.97493107214784724412481864419533e-01 1.61997493373319256804165888752323e-01 9.40985776160913230115045990942235e-03
            1.61997493373414375161800649038923e-01 7.97493107214728103038225981435971e-01 4.05093994118574940443977538961917e-02 9.40985776160714604277046646529925e-03
            7.97493107214813035099609805911314e-01 1.61997493373290002427467015877482e-01 4.05093994118969624729231782112038e-02 9.40985776160928322209286989163957e-03
            1.61997493373358253387905847375805e-01 4.05093994118448305630231232044025e-02 7.97493107214796936865752741141478e-01 9.40985776160042051985410438419422e-03
            3.86421555195467947285692389414180e-01 3.86421555195499977219952825180371e-01 2.27156889609032075494354785405449e-01 1.17630490135272481461159443938413e-02
            3.86421555195499977219952825180371e-01 2.27156889609037238031419292383362e-01 3.86421555195462840259779113694094e-01 1.17630490135089450787209131021882e-02
            2.27156889608964795979062500919099e-01 3.86421555195499977219952825180371e-01 3.86421555195535171289833442642703e-01 1.17630490135428155545893602607066e-02
            8.09012937932865439094598514202517e-01 9.54935310335075365761881016624102e-02 9.54935310336270243292133841350733e-02 1.17785733075349112464191847493566e-02
            9.54935310336080256377044861437753e-02 8.09012937932828912757088346552337e-01 9.54935310335630616052071673038881e-02 1.17785733075398656166665745104183e-02
            9.54935310335289361249877515547269e-02 9.54935310335349868404719586578722e-02 8.09012937932936049278964674158487e-01 1.17785733075352356397091924122833e-02
            2.74542523871785681954804658744251e-01 4.79840480721273610065225057041971e-02 6.77473428056087012549824066809379e-01 1.34123103715159536220014047103177e-02
            4.79840480721468107261351576653396e-02 6.77473428056089232995873317122459e-01 2.74542523871763921583522005676059e-01 1.34123103715220078069325637670772e-02
            6.77473428056023729837420432886574e-01 4.79840480721209425296613915179478e-02 2.74542523871855348449599887317163e-01 1.34123103715151834047780710079678e-02
            6.77473428056128090801735197601374e-01 2.74542523871725785422626131548896e-01 4.79840480721461237756386708497303e-02 1.34123103715211057507250558273881e-02
            2.74542523871874555307925902525312e-01 6.77473428055990978258193990768632e-01 4.79840480721344109227288754482288e-02 1.34123103715201846125593121428210e-02
            4.79840480721299630917364709148387e-02 2.74542523871823651582246839097934e-01 6.77473428056046378387122786079999e-01 1.34123103715189199991453250504492e-02
            4.05347244666806272306303071673028e-01 5.42984962234297929839499374793377e-01 5.16677930988957978541975535335951e-02 1.57144888389393368177682219766211e-02
            5.16677930988909961396160497315577e-02 4.05347244666757422493219564785250e-01 5.42984962234351553611588769854279e-01 1.57144888389387123173168703260671e-02
            4.05347244666699413340182900356012e-01 5.16677930988905728670879113906267e-02 5.42984962234409951342684053088306e-01 1.57144888389360165570352023678424e-02
            5.42984962234337564801478492881870e-01 5.16677930988870409700908226113825e-02 4.05347244666775408106218492321204e-01 1.57144888389357181845973343570222e-02
            5.16677930988952704982608565842384e-02 5.42984962234409396231171740510035e-01 4.05347244666695361026143018534640e-01 1.57144888389395519234792431007008e-02
            5.42984962234452250839922271552496e-01 4.05347244666650619038250624726061e-01 5.16677930988971301218271037214436e-02 1.57144888389394582484115403531177e-02
            1.87773861553900633936109443311580e-01 1.06814826758807768225523204819183e-01 7.05411311687291542327216120611411e-01 1.68598096079381441769751148740397e-02
            7.05411311687231146194676512095612e-01 1.87773861553885312858369616151322e-01 1.06814826758883540946953871753067e-01 1.68598096079427897664437807634386e-02
            7.05411311687196063147098357148934e-01 1.06814826758797304373516112718789e-01 1.87773861554006632479385530132276e-01 1.68598096079385049994581180499154e-02
            1.06814826758885511592822581405926e-01 7.05411311687161868277939902327489e-01 1.87773861553952592373661900637671e-01 1.68598096079438097838476551260101e-02
            1.87773861554004939389272976768552e-01 7.05411311687162645434057139937067e-01 1.06814826758832359665518652036553e-01 1.68598096079426856830352221550129e-02
            1.06814826758819994556581889355584e-01 1.87773861553926002532222128138528e-01 7.05411311687254016788983790320344e-01 1.68598096079410966763312273997144e-02
            1.19505971200887586847194654637860e-01 3.05712299064265724535260915217805e-01 5.74781729734846758006483469216619e-01 2.13872647106686346318848990222250e-02
            1.19505971200916022434412866459752e-01 5.74781729734796575925770412140992e-01 3.05712299064287429395392337028170e-01 2.13872647106708030362298700310930e-02
            5.74781729734786916985456173279090e-01 1.19505971200873140070086719788378e-01 3.05712299064339942944457106932532e-01 2.13872647106647419124048070671051e-02
            5.74781729734875623805123723286670e-01 3.05712299064210824006693201226881e-01 1.19505971200913552188183075486450e-01 2.13872647106692764795710104408499e-02
            3.05712299064345549570731463973061e-01 5.74781729734756718919186369021190e-01 1.19505971200897787021233398263576e-01 2.13872647106689364737697189866594e-02
            3.05712299064230197398472910208511e-01 1.19505971200881550009498255349172e-01 5.74781729734888169325301987555576e-01 2.13872647106650784487591465676815e-02
            5.98124574336285963482851002481766e-01 2.00937712831805004309870810175198e-01 2.00937712831909032207278187343036e-01 2.20569466368450775584086898106762e-02
            2.00937712831901760246466892567696e-01 5.98124574336253878037439335457748e-01 2.00937712831844361716093771974556e-01 2.20569466368429160929576227090365e-02
            2.00937712831833842352935448616336e-01 2.00937712831819270675737243436743e-01 5.98124574336346914726902923575835e-01 2.20569466368439395798084490252222e-02
            2.16077520000508976893982548972417e-01 3.12136025667285765816671982975095e-01 4.71786454332205285044921083681402e-01 2.30734797342044863077692440356259e-02
            3.12136025667357375201760305571952e-01 2.16077520000487938167665902255976e-01 4.71786454332154714386149407800986e-01 2.30734797342171740752725384027144e-02
            2.16077520000537232069959259206371e-01 4.71786454332146831802674569189548e-01 3.12136025667315908371790555975167e-01 2.30734797342040456880063459266239e-02
            3.12136025667380412329521277570166e-01 4.71786454332108307063720076257596e-01 2.16077520000511280606758646172239e-01 2.30734797342001113351628305281338e-02
            4.71786454332162707991926708928077e-01 3.12136025667294259022810365422629e-01 2.16077520000543032985262925649295e-01 2.30734797342167854972139195979253e-02
            4.71786454332112858978121039399412e-01 2.16077520000476058781302413080994e-01 3.12136025667411054485000931890681e-01 2.30734797342060544977915270692392e-02
            4.37657990384939132866293221013620e-01 4.37657990384801298677786007829127e-01 1.24684019230259568455920771157253e-01 2.34576234312250039848457561220130e-02
            4.37657990384825723584327761273016e-01 1.24684019230242804088248931293492e-01 4.37657990384931527838574538691319e-01 2.34576234312225198608281573342538e-02
            1.24684019230249965026757763553178e-01 4.37657990384861306232266997540137e-01 4.37657990384888728740975238906685e-01 2.34576234312236786561101098413928e-02
            3.33333333333324155489663098705932e-01 3.33333333333314052460139009781415e-01 3.33333333333361792050197891512653e-01 2.75599990173733683240797631697205e-02
            ];
        
    case  23
        
        % ALG. DEG.:   23
        % PTS CARD.:  105
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 8.7e-16
        
        xyw_bar=[
            8.78093038360354405380547859749640e-03 9.90367643677188547179923716612393e-01 8.51425939207950399634228233480826e-04 3.21914913041228582194636098634533e-04
            9.90367531421999958141100250941236e-01 8.78092162321842534256077783538785e-03 8.51546954781616516338971223376575e-04 3.21920653817614639807115350933486e-04
            2.70292764499421205373619336853608e-03 3.35914404438093713856083866176050e-02 9.63705631911196403116548481193604e-01 5.06736785510361869427908843022124e-04
            3.35909214524000021562244455708424e-02 2.70289467097205998724618325468327e-03 9.63706183876627986428786698525073e-01 5.06737628798982443420473309458885e-04
            9.16750686061702790841110299879801e-03 9.16763530514256783088278268678550e-03 9.81664857834240400791259162360802e-01 9.83996496744093127398422815588219e-04
            9.67556818255816075691200239816681e-01 8.47371766560996164674435249253293e-03 2.39694640785739626620554076907865e-02 1.67336568919417634677959672018233e-03
            8.47372006880422676267095027924370e-03 9.67556943534542246432295087288367e-01 2.39693363966535510911626261076890e-02 1.67336696041465136930437296314267e-03
            7.87819487922387061418216802621828e-03 6.76784943861999943592877571063582e-02 9.24443310734576084719549271540018e-01 2.14366616876521557874535339749400e-03
            6.76785477699999998835167502875265e-02 7.87816592909016365220420397008638e-03 9.24443286300909772279510434600525e-01 2.14367299427602794187341928022761e-03
            9.47026695504659787872014931053855e-01 4.42974541187251200091701264227595e-02 8.67585037661509211881494252338598e-03 2.15019006860308519238356161906722e-03
            4.42974755679406348440885210493434e-02 9.47026667648678599498168750869809e-01 8.67585678338078647442443980253302e-03 2.15019245491028000716315915497034e-03
            9.14424321488164104643203700106824e-01 8.17354551315039798420336580875301e-03 7.74021329986854939031459821308090e-02 2.84673146025487145555077361791518e-03
            8.17354244589089257155833934120892e-03 9.14424423403110364638735063635977e-01 7.74020341509987774841761165589560e-02 2.84673200667961588325449717729043e-03
            2.49745229274100005723013850911229e-01 3.83323243471999974829600432713050e-01 3.66931527253899991691810100746807e-01 3.08219340071967185315515713739387e-03
            3.83323264605499980373082280493691e-01 2.49745126800500011343686423970212e-01 3.66931608593999980527655679907184e-01 3.08223782093048790861611330171854e-03
            8.87685035355727181105578438291559e-01 1.03532880944553346003189631119312e-01 8.78208369971947289123193058912875e-03 3.10072567953965753939638005931556e-03
            1.03532922829663281971690480531834e-01 8.87684993183984527576058098929934e-01 8.78208398635216269667580490931869e-03 3.10072659758573853511220974610296e-03
            7.72559236181191190395711743121865e-03 1.40319099197360630526532077055890e-01 8.51955308440827452365340377582470e-01 3.48181651469703585047299476684657e-03
            1.40319242510639513588088789219910e-01 7.72559346238606541729110332994424e-03 8.51955164026974354207766282343073e-01 3.48181659209443944077988497554088e-03
            8.10459100965200018862333308788948e-01 1.80964252392600916108378328317485e-01 8.57664664219906502928836289356695e-03 3.75331288601491981984925594417746e-03
            1.80964300371699998715158130835334e-01 8.10459051533444885251356026856229e-01 8.57664809485514378906145793735050e-03 3.75331322825472228979504230039765e-03
            8.33076794868340519961691370554036e-01 8.30109396770613121308013404586745e-03 1.58622111163953360968292827237747e-01 3.95373841694603069679692097793122e-03
            8.30109071261880826497758789628278e-03 8.33076854539217759310076871770434e-01 1.58622054748163376913794309075456e-01 3.95373862424553638300617208756194e-03
            3.48407706147432008614295284587570e-02 3.48406969481684300471435733470571e-02 9.30318532437088396847002513823099e-01 4.01766723113031447994769607134913e-03
            2.74028767960759755961674954960472e-01 7.17398184794867055380507281370228e-01 8.57304724437313314666653241147287e-03 4.39817205367933495435694979391883e-03
            7.17398222477858449330767598439706e-01 2.74028730438602985408635959174717e-01 8.57304708353856526059644238557667e-03 4.39817240562447725471528769958240e-03
            2.39497685823422484796552112129575e-01 8.18591822619156410378860044829707e-03 7.52316395950386018753874850517605e-01 4.56520978577589463254238211220581e-03
            8.18591858449671701258854028537826e-03 2.39497556667719574408081939509430e-01 7.52316524747783765825204227439826e-01 4.56521068056600989870652540503215e-03
            6.88361520745116479969016864970399e-03 4.84374089268735374957941530738026e-01 5.08742295523813448099303968774620e-01 4.64108743752643642527244693951616e-03
            4.84374148569916229689624742604792e-01 6.88362329487813943973417707411500e-03 5.08742228135205643013705412158743e-01 4.64109078311248420684043125561402e-03
            4.96076777274090274794104971078923e-01 4.96076752950683486087513074380695e-01 7.84646977522623911838195454038214e-03 4.72499030887692066105199018011263e-03
            6.11293677624577647122805501567200e-01 3.80432369123821667589169237544411e-01 8.27395325160068528802526088838931e-03 4.73137342419695239303933931296342e-03
            3.80432398034445951040538602683228e-01 6.11293646653274347357864826335572e-01 8.27395531227970160159657098120078e-03 4.73137426471989790827121780125708e-03
            7.30389071352401275660781720944215e-01 8.39871797008916350091300984104237e-03 2.61212210677509548695240937377093e-01 4.77778861424359765214253670251310e-03
            8.39871686392917003727287550418623e-03 7.30389089540734115502118584117852e-01 2.61212193595336761298142391751753e-01 4.77778964215292301781490280632170e-03
            6.12852567561200767265461308852537e-01 7.54759796947155042062815866188430e-03 3.79599834469327657160420130821876e-01 4.80694212439850779988725903990598e-03
            7.54759610367729868313224628195712e-03 6.12852548458182599944166213390417e-01 3.79599855438140121322021514060907e-01 4.80694234129438405928036459613395e-03
            7.95253165133135049569279573233871e-03 3.55977382672119313511416294204537e-01 6.36070085676549323849826578225475e-01 4.99957621060644844673825915037924e-03
            3.55977487045950280464268189462018e-01 7.95253585022433380491424514957544e-03 6.36069977103825401343328849179670e-01 4.99957759252242159297985324428737e-03
            9.11023697796606413135123148094863e-01 4.37233665344507055339207113320299e-02 4.52529356689428813309561405731074e-02 5.01506596383042863723789039909207e-03
            4.37233605166329891700272014531947e-02 9.11023680744558528310506062553031e-01 4.52529587388085241528301594371442e-02 5.01506733177868417594691408112340e-03
            3.88480061834737405712658642187307e-02 9.67030908281891893318871211704391e-02 8.64448902988337097852422630239744e-01 6.24683380926671291688556664212228e-03
            9.67032117936429402860198933922220e-02 3.88479942386262128151130923470191e-02 8.64448793967730888532230437704129e-01 6.24683630623430307532428784611511e-03
            8.73226911311275888483152129992959e-02 8.73226620391871488635615605744533e-02 8.25354646829685290043698842055164e-01 7.00986545686548175038144137261042e-03
            4.21445202084233874750829329514090e-02 8.48561778910808195419690491689835e-01 1.09293700880768396288544863637071e-01 7.16681084481504693250997206632746e-03
            8.48561797496106828830875201674644e-01 4.21445420914689297675614909621800e-02 1.09293660412424248340457211270405e-01 7.16681360626749471703922367282757e-03
            8.47792133386372048775569965073373e-01 1.06743594247183659629030216819956e-01 4.54642723664442915953998181066709e-02 7.68020713700497126286137472561677e-03
            1.06743588939808814464171859981434e-01 8.47792132814562227061117027915316e-01 4.54642782456289307191354964743368e-02 7.68020917124469324616109489056726e-03
            1.83396652199136978023119581848732e-01 4.16340521608503724704242188181524e-02 7.74969295640012600934198871982517e-01 9.22619128070672166186394491660394e-03
            4.16340541166712388432280533834273e-02 1.83396519692967219228307840239722e-01 7.74969426190361576622933625912992e-01 9.22619315728200224679955709916612e-03
            7.61163225156013090177964386384701e-01 1.94159920285148612784098531847121e-01 4.46768545588382970379370817681774e-02 9.79169917866638817027791219516075e-03
            1.94159925414428363721341952441435e-01 7.61163215393799386099260573246283e-01 4.46768591917722224238218586833682e-02 9.79170099968464276163349069292963e-03
            7.57937874717249937539520487916889e-01 4.39826608586334974471121483929892e-02 1.98079464424116558074473459782894e-01 9.88163756710608298661213666491676e-03
            4.39826512394945398631840305370133e-02 7.57937824230775603062681966548553e-01 1.98079524529729877890815714636119e-01 9.88163833383549905509113386870013e-03
            3.69760535918213575912183443961112e-02 5.36318607643585854027890036377357e-01 4.26705338764592823075361138762673e-01 9.94031955096357812406981935282602e-03
            5.36318713434215554869410880201031e-01 3.69760780935089808374449660277605e-02 4.26705208472275443476462442049524e-01 9.94032428882529045521820876274433e-03
            1.00125694892088765164395169904310e-01 7.91226709354494128767498750676168e-01 1.08647595753417092190318271605065e-01 1.03590919242146847767394035599864e-02
            7.91226669352416922542658994643716e-01 1.00125755467330726511399063838326e-01 1.08647575180252350945941941517958e-01 1.03590967446462560441755584861312e-02
            3.79866714177100953109267322815867e-02 4.15741312855805678072584896654007e-01 5.46272015726484205799806659342721e-01 1.04471535719778598622076160040706e-02
            4.15741402896513068121464584692148e-01 3.79867061534608085882069872241118e-02 5.46271890950026151045904043712653e-01 1.04471625977747457902911065730223e-02
            6.50710649146250763585896947915899e-01 4.20141226713095228273431303023244e-02 3.07275228182439685831184306152863e-01 1.07432286942361586434824261004906e-02
            4.20141133438424047019665863444970e-02 6.50710564508427236063425880274735e-01 3.07275322147730345356819725566311e-01 1.07432293003460223312917065641159e-02
            4.25548444253750074439679451643315e-02 2.92062602348381217254313924058806e-01 6.65382553226243733668354707333492e-01 1.11109066517925594158588964432965e-02
            2.92062710724053598010385712768766e-01 4.25548546752879397470081812571152e-02 6.65382434600658378975879259087378e-01 1.11109080101368922627314006490451e-02
            5.38972953818032118888936565781478e-01 4.19303146900484813297538266851916e-01 4.17238992814830678135251673666062e-02 1.11672652727673668571206633259862e-02
            4.19303182848857369791062410513405e-01 5.38972909361016605522820555052022e-01 4.17239077901260246861170344345737e-02 1.11672689369570286160859851065652e-02
            6.54947200970027787469973645784194e-01 3.00735263616236814243620756315067e-01 4.43175354137353982864055979007389e-02 1.12379462472980296700031388468233e-02
            3.00735279091710550058991202604375e-01 6.54947181273061840300897529232316e-01 4.43175396352275541289600369054824e-02 1.12379490219814259094333053212722e-02
            3.75240077158565776827003901416901e-01 3.45398013075200016608334863121854e-01 2.79361909766234206564661235461244e-01 1.14850697922340844181476882113202e-02
            3.45398028278600000007259041012730e-01 3.75240069567314926679557629540795e-01 2.79361902154085073313183329446474e-01 1.14851697219190765353324934494594e-02
            9.94532168761113610777613303071121e-02 1.59830869518772511472803898868733e-01 7.40715913605116238471737233339809e-01 1.16399188051123508857997990162403e-02
            1.59830935958481357150873236605548e-01 9.94531960132452180767259619642573e-02 7.40715868028273494161339840502478e-01 1.16399213752933373272613337690018e-02
            1.79732666166714710742624561135017e-01 7.12458543092414076092211416835198e-01 1.07808790740871240920739637658698e-01 1.34741599823702505600575562993981e-02
            7.12458446194299988007969659520313e-01 1.79732772223980236114826425364299e-01 1.07808781581719775877203915115388e-01 1.34741653553579147223340228833877e-02
            1.06606567863572837540075965989672e-01 7.00170178417466426701309956115438e-01 1.93223253718960763514189693523804e-01 1.40219379005186994824327939568320e-02
            7.00170190409589277003021834389074e-01 1.06606585567697198513315015588887e-01 1.93223224022713524483663150022039e-01 1.40219382303440904397051625096537e-02
            9.93303629800770110369612098111247e-02 6.06564798479641065220846485317452e-01 2.94104838540281909864404497056967e-01 1.43763135085928119016873694135938e-02
            6.06564805252115091249720535415690e-01 9.93303896768639588144012009252037e-02 2.94104805071020936058090455844649e-01 1.43763193635679260157234082839750e-02
            1.02322354270440840640077340140124e-01 2.53338157952816500628756557489396e-01 6.44339487776742769753468564886134e-01 1.49490414531378602858158188837479e-02
            2.53338232493812542944766619257280e-01 1.02322382618884696880989793044137e-01 6.44339384887302801807607011141954e-01 1.49490461379290541510389545010185e-02
            6.16622671521702914176898957521189e-01 2.76950206057526682190683686712873e-01 1.06427122420770403632417355765938e-01 1.54502179258054206378858452808345e-02
            2.76950069310862967419950564362807e-01 6.16622790062359649176926268410170e-01 1.06427140626777383403123167227022e-01 1.54502192977834153925220661562889e-02
            9.04184571873021675214587844493508e-02 4.98152263700143593805336195146083e-01 4.11429279112554280306568443847937e-01 1.57015508544004790680848060446806e-02
            4.98152276724787179684739157892182e-01 9.04185045149451249724137369412347e-02 4.11429218760267612076120258279843e-01 1.57015536977641104177116915252554e-02
            9.28231860168124284227175735395576e-02 3.73841851690821957987509449594654e-01 5.33334962292365655223136400309158e-01 1.59595776511899135885474976248588e-02
            3.73841869922919289859208902271348e-01 9.28232584789866893570930983514700e-02 5.33334871598094006905910191562725e-01 1.59595834189084351262266636695131e-02
            2.52167884040700007730606557743158e-01 2.52168092569632074084751138798310e-01 4.95664023389667918184642303458531e-01 1.60714962031055473568041946919038e-02
            5.08750021870755353425863631855464e-01 3.90558054432983914328048058450804e-01 1.00691923696260732246088309693732e-01 1.65197800693986304731986081151263e-02
            3.90557911673060198509688234480564e-01 5.08750143766099505171496275579557e-01 1.00691944560840296318815489939880e-01 1.65197815914440923612538369980030e-02
            1.70614146909608382918221991531027e-01 5.26673803955435726464884282904677e-01 3.02712049134955862861318109935382e-01 1.78084547794238774731656604899399e-02
            5.26673776131196036409676253242651e-01 1.70614225753710269106733221633476e-01 3.02711998115093694483590525123873e-01 1.78084638027084067590788407642322e-02
            3.48758152762900008880819768819492e-01 2.58805508488593516425169127614936e-01 3.92436338748506474694011103565572e-01 1.82870594998830703070691328093744e-02
            2.58805359601724960239010897566914e-01 3.48758349170337211830883461516351e-01 3.92436291227937772418954409658909e-01 1.82870757602350102188903946398568e-02
            1.69661455805267119556489774367947e-01 3.01352218396394311294983481275267e-01 5.28986325798338485881799897470046e-01 1.82988823495229060989153424543474e-02
            3.01352180687456117080813555730856e-01 1.69661596321832502942683618130104e-01 5.28986222990711296709775979252299e-01 1.82989026944562344079159288412484e-02
            2.58020240975833925478610808568192e-01 4.58474177447841080290658055673703e-01 2.83505581576325049741882367015933e-01 1.84972840056980211698167693157302e-02
            4.58474086019770632471903581972583e-01 2.58020381901117201728368399926694e-01 2.83505532079112110288576786842896e-01 1.84972887529235956172168897637675e-02
            1.84889868349759234344986680298462e-01 1.84889870455055865816618165808904e-01 6.30220261195184816571668307005893e-01 1.87026811893742268833040753861496e-02
            6.13074033846504717892855751415482e-01 1.92161199406937599487577017498552e-01 1.94764766746557682619567231085966e-01 1.87775129158746188762219020418343e-02
            1.92161175099370584717917154193856e-01 6.13074039838860329609815380536020e-01 1.94764785061769085672267465270124e-01 1.87775156264800707761608578039159e-02
            4.18054116059869362853618213193840e-01 1.65061333641641855285442375134153e-01 4.16884550298488698594212564785266e-01 1.94443846743210246863498014135985e-02
            1.65061264203588409227307920446037e-01 4.18054119924393174922983007490984e-01 4.16884615872018360338557840805151e-01 1.94443854170879421028317324271484e-02
            5.15920573962530881750865319190780e-01 2.98271900522946442624316887304303e-01 1.85807525514522675624817793504917e-01 1.96352821774104686902706617956937e-02
            2.98271893574985735853033474995755e-01 5.15920553436231665855871142412070e-01 1.85807552988782598291095382592175e-01 1.96352901258759413027910056825931e-02
            4.09889460233993307980426834546961e-01 4.09889431779187407567377476880210e-01 1.80221107986819284452195688572829e-01 1.99383439915744829773469604106140e-02
            ];
        
    case  25
        
        % ALG. DEG.:   25
        % PTS CARD.:  120
        % NEG. W.  :    0
        % OUT PTS. :    3
        % M.E.INF. : 7.0e-16
        
        xyw_bar=[
            8.28815950327991060597732086989708e-03 9.84820276886916201597443887294503e-01 6.89156360980391902160135941812769e-03 7.43670892954301199738453842513763e-04
            4.61842203024124764443314461459522e-01 5.38157796975900049041285910789156e-01 -2.47579734491409908514469861984253e-14 7.44451763099710948565213008976116e-04
            7.10664412391408435720752123643251e-03 8.08423613900631152229259157593333e-03 9.84809119737079630141352026839741e-01 7.50297218993128782471568882783686e-04
            9.84761314169917412186805449891835e-01 7.00157551338421221126528948275336e-03 8.23711031669837473456752263700764e-03 7.52960415636258375941813891785159e-04
            5.37444786904896942836273865395924e-01 4.62555213095117545574197492896928e-01 -1.44884104713582928525283932685852e-14 7.65943435751058982960404186712822e-04
            5.14426194399587329140975825306526e-15 4.88767688014002554819370516270283e-01 5.11232311985992282643564976751804e-01 1.15163172435659138387320510332756e-03
            4.91413192936144105349427491091774e-01 -2.62191835077036522842371265891708e-14 5.08586807063882151425104893860407e-01 1.18245335207546207675899374578421e-03
            7.03459370203730286613685862562306e-03 9.57415805369697281435037439223379e-01 3.55496009282654235050813440466300e-02 1.43755718053232943443031643937502e-03
            9.56473471422832233201916096732020e-01 3.64655449485063384851279977283411e-02 7.06098362866142831295590553963848e-03 1.49312443676099474570984071419844e-03
            3.70198792044907007414700217395875e-02 7.09085771655651366657258094505778e-03 9.55889263078952833296852986677550e-01 1.51920813685521636886288643353282e-03
            1.02412454274703856871298057740205e-01 8.93612559493700997492737769789528e-01 3.97498623159520114711540372809395e-03 1.60462298441627960500499572304989e-03
            5.92806581150900013810201016895007e-01 4.94517055997421069862829412500105e-03 4.02248248289125753807127239269903e-01 1.85147992173100755033954278871988e-03
            5.09484223714734389687475157870722e-03 9.96676659188651253851176647913235e-02 8.95237491843987598372223146725446e-01 1.87035930174580806821527101391212e-03
            8.15620236892535682704519217622874e-03 4.15561148783834197195652393475029e-02 9.50287682752691265086752991919639e-01 1.92262716115360098381725784122409e-03
            4.24936107568289617164580818098329e-02 9.49486526035189015715332061517984e-01 8.01986320798198093484643322881311e-03 1.93353893342305700048244787581098e-03
            9.49554350084383824714961974677863e-01 8.17945072920253993253858482148644e-03 4.22661991864136388219463924542652e-02 1.95962775887665967278206125001816e-03
            8.93278747123910199512408780719852e-01 5.32243262623153211288151709368321e-03 1.01398820249858273578880130116886e-01 1.97866413439398095702737023771078e-03
            6.93176129272652947338473339300435e-03 9.06540102043311923907253913057502e-01 8.65281366639615301394883317698259e-02 2.20161258618352222160563513853049e-03
            9.03583903066471405551851603377145e-01 8.94771171076942650746133267603000e-02 6.93897982583432937353506986255525e-03 2.29535540867448163357411594631685e-03
            9.05665738208722287394536465399142e-02 7.05253420051081634373213447020134e-03 9.02380891978617039050902803865029e-01 2.35118347173016986259574956363849e-03
            8.39293327869924249418076556139567e-03 6.66317993111099959868681708030635e-01 3.25289073610200762942668006871827e-01 2.50074219087919463744285053508065e-03
            6.26124568607099973860385944135487e-01 9.21975831529366594840890769546604e-03 3.64655673077606379273163383913925e-01 2.61939150778003806280258558558671e-03
            6.28015929789052865239717959866539e-03 8.33520746052701988482169781491393e-01 1.60199094649407514090455606492469e-01 2.72110520462297451882438537040798e-03
            8.27253925736676443136730085825548e-01 1.66513493933040795935696110063873e-01 6.23258033028276092757380411057966e-03 2.84656244562468338510363707882789e-03
            6.20058753533867922785871584778761e-03 7.42469325522900014391325385076925e-01 2.51330086941761288166219401318813e-01 2.95537114945203585603117346636282e-03
            1.67690031118506527318956500494096e-01 6.57177435282130240068454440915957e-03 8.25738194528672142524783339467831e-01 2.98439838434796846783347845644130e-03
            7.19935306956684839541082965297392e-01 6.43545349617986699175542497641800e-03 2.73629239547135305610225941563840e-01 3.36310951436107063994285226726788e-03
            2.74974009023742749224794579276931e-01 7.18529612071858725386164223891683e-01 6.49637890439858090019242808921263e-03 3.41539243122544093969605860650063e-03
            7.92575820045364476396088093679282e-03 1.76641137471416409532309899077518e-01 8.15433104328129987337092643429060e-01 3.47656295559924024413112064735287e-03
            6.99812207516695237302650767219347e-03 2.70476725400404682808641609881306e-01 7.22525152524428393441269236063818e-01 3.62301353211745644916641495569820e-03
            8.12524877326335270844026581471553e-01 8.22995332101647721068093943586064e-03 1.79245169352648253680015955069393e-01 3.63640948064765986286195342813699e-03
            7.35369699704123847699976934677579e-03 5.93416787545230528522210988739971e-01 3.99229515457728223459810124040814e-01 3.65044654232891908549052395471790e-03
            7.28366593541109663689780973072629e-01 2.64881755375208860492364237870788e-01 6.75165108368147581785478905658238e-03 3.68023333878022412823938225301390e-03
            1.80064230456454726914117259184422e-01 8.11584897668173987916873102221871e-01 8.35087187537131292458525422262028e-03 3.70599616275578934432299860191051e-03
            2.65810246776189362805098426179029e-01 6.85535254287219369961903225885180e-03 7.27334400680938464311964253283804e-01 3.74461071680228701000214819316625e-03
            7.08923645195805307572811670979718e-03 3.75763265974363824817316981352633e-01 6.17147497573678149862530517566483e-01 3.93020336298502501876095394095501e-03
            3.77405430204347802636277720012004e-01 6.14857353375700022013461421011016e-01 7.73721641995217535026085897698067e-03 3.93108632115824179614760325307543e-03
            3.69649608667726101152162243579369e-02 9.21079230289327588465653207094874e-01 4.19558088438998222358122802688740e-02 4.02531805328896607337751945010496e-03
            9.20319410980492857987655952456407e-01 4.26025082113667408201251873833826e-02 3.70780808081404011922188601602102e-02 4.07214302363667277062653226948896e-03
            4.25477806431233118722445851744851e-02 3.72689941794178275502602559754450e-02 9.20183225177458763432980504148873e-01 4.07394020759452996083282982908713e-03
            6.19127839498267773521433809946757e-01 3.72405571380885980214969777080114e-01 8.46658912084624626359641297312919e-03 4.62220733060178263357231642771694e-03
            3.76269720917775551960460234113270e-01 8.14364220110922777673412298327094e-03 6.15586636881115256691998638416408e-01 4.73373175827357297457131934947938e-03
            9.56111149690206385542978750891052e-02 8.77109837260114377777142635750351e-01 2.72790477708649836685594891605433e-02 4.85661050683420039697502090803027e-03
            3.02473410377312491859047582920539e-02 9.43858903393274828452774727338692e-02 8.75366768622941160415962258412037e-01 4.98767905754648761124103728548107e-03
            8.73990569175390308309658848884283e-01 3.13198990882912128852488820029976e-02 9.46895317363184718661983652054914e-02 5.16839018366951636479500464815828e-03
            8.60413373495807265456392087799031e-01 1.04901978204618509527001890546671e-01 3.46846482995742250166060216542974e-02 5.61316385828435765459332174032170e-03
            3.47307852352399665241122761472070e-02 8.60985646288632278100294570322149e-01 1.04283568476127741497805345716188e-01 5.71545593722535584524369056680371e-03
            1.04360660834317178768948508604808e-01 3.57152881004371841600075754286081e-02 8.59924051065245609315468300337670e-01 5.77752837433548847856368269049199e-03
            7.79762282475343737608852734410902e-01 1.87231819926519243990625795959204e-01 3.30058975981370184005214696298935e-02 6.77879284784793905566102623083680e-03
            1.85865164256267152709245493724666e-02 4.83439767879348392121130473242374e-01 4.97973715695024909955179737153230e-01 6.79924814498323887984598812295189e-03
            3.24585286618077825671058178613748e-02 7.78347491604180818924874074582476e-01 1.89193979734011419324701819277834e-01 6.88774069183051632081493664827576e-03
            8.37129390115686100592995444458211e-01 8.04060570156074766456910651868384e-02 8.24645528687064227613134903549508e-02 6.89805079711462015845668815927638e-03
            8.36602075314564763708347072679317e-02 8.42141481705122707523969438625500e-01 7.41983107634207605940446228487417e-02 6.92044199518837468282361768956434e-03
            7.84070242500509301253330818326504e-02 8.49927089144874164938769922628126e-02 8.36600266835461625625214310275624e-01 7.03170099884228589487422667048122e-03
            4.92923864845802028877841394205461e-01 4.89285591471027347942879259790061e-01 1.77905436831705676681281147466507e-02 7.04957255044685350553468339285246e-03
            1.87063758407251617965627588091593e-01 3.45210858281027530014561932603101e-02 7.78415155764645594338446699111955e-01 7.10020559953956301862953992554139e-03
            4.89263696702512351777158983168192e-01 1.90774755076611041149092784507957e-02 4.91658827789826557985719546195469e-01 7.22592122586531489941696904111268e-03
            4.01982618371916830546375365429412e-02 1.69114318710903849263260667612485e-01 7.90687419451904460743207891937345e-01 7.51229898194268313343924248215444e-03
            7.89425927886497902363771572709084e-01 4.12206731483492141987490242627246e-02 1.69353398965152890376373306935420e-01 7.64089020609124563998548751442286e-03
            1.68626045642937899549096414375526e-01 7.89486064058546044286401865974767e-01 4.18878902985160284089261040207930e-02 7.77753620844678309614872446786649e-03
            3.75090191317404897386467155229184e-01 5.89531827201280833605778752826154e-01 3.53779814813142134966028606868349e-02 8.22854430002309573943808373996944e-03
            3.56362876879900564164316278947808e-02 3.68125621769863742294148778455565e-01 5.96238090542146159656056170206284e-01 8.26378797863224802411341585184346e-03
            5.88754816480384635113409785844851e-01 3.59968962540969730778961377382075e-02 3.75248287265518398747587980324170e-01 8.34237772252478977652589975377850e-03
            3.73308082181670955468355543871439e-02 6.79070467353293794054991394659737e-01 2.83598724428539061825915723602520e-01 8.37046564926244816839950857456643e-03
            2.82076999337395051270505064167082e-01 3.73639992361457645708711083898379e-02 6.80559001426459198036411635257537e-01 8.43373316802667472025234474131139e-03
            6.81927760331965671092291358945658e-01 2.80333034572545214402339297521394e-01 3.77392050954891145053693435329478e-02 8.44411150824622008803643780083803e-03
            3.74938324381651408301330263839191e-02 2.63401618001446569117263152293162e-01 6.99104549560388255358134301786777e-01 8.60435563454054093412981529809258e-03
            6.98407920412669636078817347879522e-01 3.64154673321932242924248157578404e-02 2.65176612255137111873182220733725e-01 8.73405341320861519638363290596317e-03
            2.65439089407868589187700081311050e-01 6.98071743619317341966734602465294e-01 3.64891669728140133344140849658288e-02 8.83319498072075368066968792390981e-03
            1.42984844080011913591832239944779e-01 7.61225461845349782308289832144510e-01 9.57896940746383318554535435396247e-02 9.14838107374081783984554050448423e-03
            7.62355400764634705268463221727870e-01 9.43741220275100267667056641585077e-02 1.43270477207855267964831114113622e-01 9.17884262293319243275480090460405e-03
            9.34222022748698310845583137052017e-02 1.47979983683175969622070056175289e-01 7.58597814041954143782220398861682e-01 9.31962847605539122797235762618584e-03
            5.75900447992329755919627132243477e-01 3.82132964169802080700577562311082e-01 4.19665878378681633797953054454410e-02 9.48905302949344317542390569997224e-03
            3.82242733252505628716022556545795e-01 4.26716362301194362949274818674894e-02 5.75085630517375046011352424102370e-01 9.59239612891127531313362197806782e-03
            4.11414081674571288638198041098804e-02 5.71808287443200891075889558123890e-01 3.87050304389341959243608926044544e-01 9.70402210221972538750634384996374e-03
            8.02462538378740236622022052870307e-02 7.70220438204229251866195227194112e-01 1.49533307957896766104965990962228e-01 9.73600360963561110849706636827250e-03
            7.62522981940991995131184921774548e-01 1.55942057736158296021855562685232e-01 8.15349603228497088469595155402203e-02 1.00427540247364031206211976154918e-02
            1.52494144513048351941009173060593e-01 8.42965421322266761361419185050181e-02 7.63209313354724971922848908434389e-01 1.00836954666030220034711462062660e-02
            6.22159195833349928927979988202424e-02 4.53818131887360642906514840433374e-01 4.83965948529304412772944488096982e-01 1.10871081380639700114665302521644e-02
            1.10953903607625831151395345841593e-01 4.58601407117075587915877576961066e-01 4.30444689275298608688302692826255e-01 1.14851220254180742819105631724597e-02
            4.57562721205716171901656252885005e-01 4.79531356021019539159766509328620e-01 6.29059227732642889385772377863759e-02 1.16732558699460895607069588209015e-02
            4.32286513637400005016786508349469e-01 1.23059123747196408338311357510975e-01 4.44654362615403586644902134139556e-01 1.17441567669165827098032295339181e-02
            5.86500285024055267868448027002160e-01 8.34119779793435350834229780048190e-02 3.30087736996601210925916802807478e-01 1.20341049508797190426001577634452e-02
            8.69359250817794138033178796831635e-02 6.75567701335066272960716560191941e-01 2.37496373583154340991541175753810e-01 1.20455396476600896937814155762680e-02
            9.29594906936006132314176397812844e-02 2.32650089272664062356454905966530e-01 6.74390420033735282778764030808816e-01 1.22838524740656821615525373658784e-02
            6.66193214145382905932990524888737e-01 2.44829400740542629799634255505225e-01 8.89773851140744642673752196060377e-02 1.23268157859734474313118113286691e-02
            4.78030636222676108548768070249935e-01 6.61749044834882976795498166211473e-02 4.55794459293835663160621152201202e-01 1.23378265025939626298168150242418e-02
            4.37221529457722968281530029344140e-01 4.44214558524454405930015354897478e-01 1.18563912017822625788454615758383e-01 1.24852301355029247920125357040888e-02
            6.77922450466891035603111959062517e-01 9.29096534576604599076077306563093e-02 2.29167896075448518367068118095631e-01 1.25013272040768209925065690413248e-02
            2.42343125565994327308416700361704e-01 8.89793655129190036445763212213933e-02 6.68677508921086682924794786231359e-01 1.25245434713039550250757159233217e-02
            2.28892542030518980622488811604853e-01 6.78005308167170972488690949830925e-01 9.31021498023100191332446229353081e-02 1.25468125062737391461897829003647e-02
            3.31506504995856432493894772051135e-01 5.84738155974041440288146986858919e-01 8.37553390301021272179582410899457e-02 1.25741038113175284535172337996300e-02
            3.42420052660672158406640619432437e-01 5.13924572273599999938653581921244e-01 1.43655375065727786143554567388492e-01 1.27505145223740755378027955657672e-02
            8.62630046474948397605686523093027e-02 3.34097624923409219910297451860970e-01 5.79639370429095968084709511458641e-01 1.28272255989636228823425057044005e-02
            5.11318894663500000952183199842693e-01 1.38015472055394355921364990535949e-01 3.50665633281105670882027425250271e-01 1.28987375314812881960113699619797e-02
            1.53897784100073625745963568078878e-01 6.78806261956247491973215346661163e-01 1.67295953943678910036396700888872e-01 1.35003876996600546966931233328069e-02
            6.77995134847147129519839836575557e-01 1.66335892526875278596421026122698e-01 1.55668972625977591883739137301745e-01 1.37215768421760780926277334401675e-02
            1.66460046941077521864826849196106e-01 1.58221450484884951226405291890842e-01 6.75318502574037582419919090170879e-01 1.38536200743844453014963491455092e-02
            9.50910318887985689384478860119998e-02 5.66659033254306332416661007300718e-01 3.38249934856895140278254530130653e-01 1.39142207681962164211864063645407e-02
            3.43604813671150643994423035110231e-01 9.78960873456868746478320986170729e-02 5.58499098983162411968805827200413e-01 1.43603690552460572932291782421999e-02
            5.56041702536571924575525827094680e-01 3.46891782094658818191845739420387e-01 9.70665153687692572326284334849333e-02 1.44413417478114741027139800166879e-02
            1.45240402951303859691023490086081e-01 3.59953449105217593295691358434851e-01 4.94806147943478491502133920221240e-01 1.46651364879301707166980861529737e-02
            1.61968515623819420135021118767327e-01 5.81013137333045603938330714299809e-01 2.57018347043134975926648166932864e-01 1.59451439778373373068465213009404e-02
            5.80016484426167822086028991179774e-01 2.56067464067228411916943286996684e-01 1.63916051506603765997027721823542e-01 1.59541830142902359146450663729411e-02
            2.45020122328836575942290210150531e-01 5.88146955210173794093009291827912e-01 1.66832922460989574453549266763730e-01 1.60469480164698169444292830121412e-02
            2.55762189179423060636509035248309e-01 1.65224406504668047679373898972699e-01 5.79013404315908863928541450150078e-01 1.60809304389862295359137078776257e-02
            2.20523998551101912957150830152386e-01 3.49650746610609075570863524262677e-01 4.29825254838288983716410029956023e-01 1.61212063766782189022652715948425e-02
            4.94018311128501474982499530597124e-01 2.54944844845292173385331579993363e-01 2.51036844026206351632168889409513e-01 1.63536223210394848770476983190747e-02
            2.53157068979825539134509426730801e-01 2.54336911501687523706749516350101e-01 4.92506019518486992669892288176925e-01 1.64973158347298254344792667325237e-02
            5.84689111635679714673585749551421e-01 1.66660391647927841329845932705211e-01 2.48650496716392443996568317743368e-01 1.65914048012727743941496783008915e-02
            1.66033360227843096446065374038881e-01 2.52324019170541480328040506719844e-01 5.81642620601615423225894119241275e-01 1.67428581325510909827425365392628e-02
            2.50542629246117087848944038341870e-01 4.95900762752834678437352522450965e-01 2.53556608001048289224854670464993e-01 1.67734236396032236615649679833950e-02
            3.51933680218193378763658074603882e-01 1.80538036780004129688848024670733e-01 4.67528283001802491547493900725385e-01 1.68524521493906530300588286763741e-02
            3.50266883541880902352971816071658e-01 4.35858232988066807145344228047179e-01 2.13874883470052234990532724623336e-01 1.70180731383460029371157418154326e-02
            4.40089248551157830036117957206443e-01 2.12057610494083004226695265970193e-01 3.47853140954759165737186776823364e-01 1.71232617661575166334575470727941e-02
            4.68085547154612702236420318513410e-01 3.55268157077371138363730551645858e-01 1.76646295768016159399849129840732e-01 1.72764408625364594696005582363796e-02
            1.77023776394722193172626134582970e-01 4.67035292226636644663528841192601e-01 3.55940931378641134408269408595515e-01 1.78391437851419809579311959168990e-02
            3.90092077950076832060233300580876e-01 3.32315281929987604847553939180216e-01 2.77592640119935563092212760238908e-01 1.82328112507940398590022823555046e-02
            2.80584777412015762276809027753188e-01 3.89804117668033467403176928200992e-01 3.29611104919950714808862812787993e-01 1.82586354353242372172694274468085e-02
            3.36152334743991199061952102056239e-01 2.77850004435603759045392280313536e-01 3.85997660820405041892655617630226e-01 1.85962405508785788565973007280263e-02
            ];
        
        
    otherwise
        xyw=[];
end




pointset_stats=[
    2    3 2.12e-16    0    0
    4    6 2.22e-16    0    0
    5   10 1.77e-13    0    0
    7   15 4.59e-13    0    0
    9   21 3.45e-16    0    0
    11   28 4.34e-16    0    0
    13   36 5.46e-16    0    0
    14   45 4.53e-13    0    0
    16   55 7.82e-13    0    0
    18   66 4.77e-16    0    0
    20   78 6.04e-16    0    0
    21   91 5.97e-16    0    0
    23  105 8.74e-16    0    0
    25  120 6.97e-16    0    3
    ];

