
function [xyw,pointset_stats]=set_taylor_wingate_bos_C_standard(deg)

%--------------------------------------------------------------------
% CUBATURE RULE ON THE SIMPLEX [0 0; 0 1; 1 0] WITH ADE EQUAL TO deg.
% THE CUBATURE RULE IS DESCRIBED IN STANDARD FORM.
%--------------------------------------------------------------------
% INPUT:
% deg: algebraic degree of precision (ADE) of the rule.
%--------------------------------------------------------------------
% OUTPUT:
% xyw: MATRIX OF NODES xyw(:,1:2) AND WEIGHTS xyw(:,3).
%      REMARK: The sum of the weights makes 1/2.
%
% pointset_stats:
%    statistics of the pointset stored in a matrix. Each rows contains
%    degree, cardinality of the full pointset, moment error w.r.t. orthn.
%    Dubiner basis,number of points with negative weights, number of
%    points outside the domain.
%--------------------------------------------------------------------
% SOURCE:
%  Mark A. Taylor, Beth A. Wingate, Len P. Bos,
%  "Several new quadrature formulas for polynomial integration in the triangle",
%  unpublished note, see arXiv.
%--------------------------------------------------------------------
% | DEG | CARD | W- | PT.O|   M.E.   |Q.|
%--------------------------------------------------------------------
% |   2 |   3 |   0 |   0 | 2.12e-16 |PI|
% |   4 |   6 |   0 |   0 | 2.22e-16 |PI|
% |   5 |  10 |   0 |   0 | 1.77e-13 |PI|
% |   7 |  15 |   0 |   0 | 4.59e-13 |PI|
% |   9 |  21 |   0 |   0 | 3.45e-16 |PI|
% |  11 |  28 |   0 |   0 | 4.34e-16 |PI|
% |  13 |  36 |   0 |   0 | 5.46e-16 |PI|
% |  14 |  45 |   0 |   0 | 4.53e-13 |PI|
% |  16 |  55 |   0 |   0 | 7.82e-13 |PI|
% |  18 |  66 |   0 |   0 | 4.77e-16 |PI|
% |  20 |  78 |   0 |   0 | 6.04e-16 |PI|
% |  21 |  91 |   0 |   0 | 5.97e-16 |PI|
% |  23 | 105 |   0 |   0 | 8.74e-16 |PI|
% |  25 | 120 |   0 |   3 | 6.97e-16 |P0|
%--------------------------------------------------------------------
% LEGEND:
%--------------------------------------------------------------------
% DEG  : algebraic degree of precision
% CARD : cardinality of the pointset
% W-   : negative weights
% PT.O : points outside the simplex [0 0; 1 0; 0 1].
% M.E. : maximum moment error in inf norm.
% Q.: quality of the rule.
%        PI: pos. weights, points in simplex
%        NI: some neg. weights, points in simplex
%        P0: pos. weights, some points out of the simplex
%        N0: some neg. weights, some points out of the simplex
%--------------------------------------------------------------------


switch deg
    
    case   2
        
        % ALG. DEG.:    2
        % PTS CARD.:    3
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 2.1e-16
        
        xyw=[
            1.66666666666666712925959359381523e-01 6.66666666666699936349971267190995e-01 1.66666666666644480709891240621801e-01
            6.66666666666700047372273729706649e-01 1.66666666666666601903656896865868e-01 1.66666666666644425198740009363974e-01
            1.66666666666699964105546882819908e-01 1.66666666666699964105546882819908e-01 1.66666666666711121846944365643139e-01
            ];
        
    case   4
        
        % ALG. DEG.:    4
        % PTS CARD.:    6
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 2.2e-16
        
        xyw=[
            9.15762135097877988387793379843060e-02 9.15762135097891449841966959866113e-02 5.49758718276768462707160267655127e-02
            8.16847572980492264704821536724921e-01 9.15762135097538954031648472664529e-02 5.49758718276457947204960419185227e-02
            9.15762135097846763365225797315361e-02 8.16847572980460179259409869700903e-01 5.49758718276601790475588416029495e-02
            1.08103018168073619631464055146353e-01 4.45948490915999973260852584644454e-01 1.11690794839008081251741089090501e-01
            4.45948490915999973260852584644454e-01 1.08103018168070386106904834377929e-01 1.11690794839005846927904030962964e-01
            4.45948490915999973260852584644454e-01 4.45948490915933359879375075252028e-01 1.11690794839003279537159585288464e-01
            ];
        
    case   5
        
        % ALG. DEG.:    5
        % PTS CARD.:   10
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.8e-13
        
        xyw=[
            0.00000000000000000000000000000000e+00 1.00000000000000000000000000000000e+00 6.56780248759999962793187933129957e-03
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 6.56791530169999974303474132852898e-03
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 6.85409869000000015332174641002894e-03
            2.67327353118499977391309130325681e-01 6.72819921871000037327803511288948e-01 5.87095966455749990187129583318892e-02
            6.72817552946100017052799557859544e-01 2.67328859948199992757622567296494e-01 5.87103059567000012952675547239778e-02
            6.49236350053999933251702714187559e-02 6.71653011149400014367927269631764e-01 6.20062948278500003507218707454740e-02
            6.71649853904199956033949092670809e-01 6.49251690029000044779294853469764e-02 6.20076230630249997100023051643802e-02
            6.54032456799999994956351656583138e-02 2.69378936645299993379865099996096e-01 6.29651151382250029442388949973974e-02
            2.69376706914000008108445172183565e-01 6.54054874919000051036732656939421e-02 6.29665133414499950115228443792148e-02
            3.38673850389600017418700872440240e-01 3.38679989302699990805223251300049e-01 1.12644734547849997841062474890350e-01
            ];
        
    case   7
        
        % ALG. DEG.:    7
        % PTS CARD.:   15
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.6e-13
        
        xyw=[
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 2.56395435230000016957330188915876e-03
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 2.56395435230000016957330188915876e-03
            0.00000000000000000000000000000000e+00 1.00000000000000000000000000000000e+00 2.56395435230000016957330188915876e-03
            7.83965665101200026576577783998800e-01 4.21382841641999988935296528325125e-02 2.79011761661750005980575650710307e-02
            1.73896050734500012824312875636679e-01 7.83965665101200026576577783998800e-01 2.79011761661750005980575650710307e-02
            1.73896050734500012824312875636679e-01 4.21382841641999988935296528325125e-02 2.79011761661750005980575650710307e-02
            4.21382841641999988935296528325125e-02 1.73896050734500012824312875636679e-01 2.79011761661750005980575650710307e-02
            7.83965665101200026576577783998800e-01 1.73896050734500012824312875636679e-01 2.79011761661750005980575650710307e-02
            4.21382841641999988935296528325125e-02 7.83965665101200026576577783998800e-01 2.79011761661750005980575650710307e-02
            4.74388086175200007765795362502104e-01 4.74388086175200007765795362502104e-01 4.19943898833749981736040979285463e-02
            4.74388086175200007765795362502104e-01 5.12238276497000016851401937856281e-02 4.19943898833749981736040979285463e-02
            5.12238276497000016851401937856281e-02 4.74388086175200007765795362502104e-01 4.19943898833749981736040979285463e-02
            2.38561530018099987593416244635591e-01 5.22876939963899944885383774817456e-01 6.63059700986499966912646186756319e-02
            5.22876939963899944885383774817456e-01 2.38561530018099987593416244635591e-01 6.63059700986499966912646186756319e-02
            2.38561530018099987593416244635591e-01 2.38561530018099987593416244635591e-01 6.63059700986499966912646186756319e-02
            ];
        
    case   9
        
        % ALG. DEG.:    9
        % PTS CARD.:   21
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.5e-16
        
        xyw=[
            4.51890097843806529120946890998312e-02 4.51890097843734711569041451184603e-02 1.29967855161600056440596162588008e-02
            4.51890097843737903460237248509657e-02 9.09621980431245868992107261874480e-01 1.29967855161600229912943760268718e-02
            9.09621980431248200460458974703215e-01 4.51890097843794594223432170565502e-02 1.29967855161593568574796009329475e-02
            7.47512472733889321574451969354413e-01 3.04243617288123165132152081469030e-02 1.76758525445948611043434084422188e-02
            2.22063165537333934151931202904962e-01 3.04243617288317072522296768966044e-02 1.76758525446061610930659213636318e-02
            7.47512472733836363936177349387435e-01 2.22063165537328827126017927184876e-01 1.76758525446075349940588949948506e-02
            2.22063165537305123864442180092738e-01 7.47512472733883548414723918540403e-01 1.76758525445950206989031983084715e-02
            3.04243617288244561081000938429497e-02 7.47512472733840471761368462466635e-01 1.76758525446024418459334270892214e-02
            3.04243617288046282187696789378606e-02 2.22063165537305429175773952010786e-01 1.76758525445914957408000134364556e-02
            1.36991201264899997713797574760974e-01 2.18290070971362881735444716468919e-01 2.27347690238328889456020931447711e-02
            6.44718727763699983590583997283829e-01 2.18290070971372207608851567783859e-01 2.27347690237963348525163098656776e-02
            1.36991201264899997713797574760974e-01 6.44718727763739174463353265309706e-01 2.27347690238005120666464620171610e-02
            2.18290070971400934629613743709342e-01 6.44718727763699983590583997283829e-01 2.27347690238199375001304503030042e-02
            2.18290070971399990940042812326283e-01 1.36991201264899997713797574760974e-01 2.27347690237876369490077621549062e-02
            6.44718727763712418088459799037082e-01 1.36991201264899997713797574760974e-01 2.27347690238204336310445796698332e-02
            3.69603304333801299663697648156813e-02 4.81519834783289857327304162026849e-01 2.58086012845117930558647145744544e-02
            4.81519834783337374872758118726779e-01 3.69603304333760152022847478292533e-02 2.58086012845094754653008095601763e-02
            4.81519834783297573377325306864805e-01 4.81519834783325717530999554583104e-01 2.58086012845098571044655244577370e-02
            4.03603979817900027082799851996242e-01 1.92792040364117206463490106216341e-01 4.70400367291777671274743966023379e-02
            4.03603979817979185984455625657574e-01 4.03603979817903135707268802434555e-01 4.70400367291770038491449668072164e-02
            1.92792040364124867002360019796470e-01 4.03603979817978575361792081821477e-01 4.70400367291794532786930460588337e-02
            ];
        
    case  11
        
        % ALG. DEG.:   11
        % PTS CARD.:   28
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.3e-16
        
        xyw=[
            2.11168130454084196363955981711748e-14 9.45170445017362803596938647388015e-01 2.85206235081708527751009896178402e-03
            9.45170445017299964973744863527827e-01 4.90750005846733871211117897806156e-14 2.85206235082173867323440674681478e-03
            9.28900240571878477169320831308141e-01 6.85505797224144203250162377116794e-02 3.31728214298547020855156830521082e-03
            6.85505797224479906937233408825705e-02 9.28900240571737367822890973911854e-01 3.31728214300055059340910368348432e-03
            2.43268355615265073621511504597947e-02 2.43268355615932664604006419040161e-02 3.89664433374836824061637763350063e-03
            1.27966283533483604628600005526096e-01 2.77838749488136006748906225993778e-02 1.02068695107132602351018846320585e-02
            2.77838749488144090560304277914838e-02 1.27966283533699987096099448535824e-01 1.02068695107210630212968283103692e-02
            2.87083428360431214265702237753430e-02 7.49834758865739803290750842279522e-01 1.44962416278996992363703455453106e-02
            7.49834758865634110058806527376873e-01 2.87083428360427571346402686458532e-02 1.44962416279075488600991405974128e-02
            7.22800790970692719206169840617804e-01 2.49760206238569426595574896055041e-01 1.50346311915747505444373999239360e-02
            2.49760206238600013239903319117730e-01 7.22800790970658635359313848312013e-01 1.50346311915782043788780697468610e-02
            8.65562992839277733914116197411204e-02 8.32551385699644819204934265144402e-01 1.56318472108221487304469121681905e-02
            8.32551385699719315169886613148265e-01 8.65562992839324085725394297696766e-02 1.56318472108313670509982529210902e-02
            3.06161915767199976912138481566217e-01 3.03526617490630160056674924362596e-02 1.59921080375974092313562380240910e-02
            3.03526617490652954323149259607817e-02 3.06161915767456771497734280274017e-01 1.59921080375972635145842559722951e-02
            4.86861059504704873912572793415166e-01 4.86861059504661464192309949794435e-01 1.65331468040212439807312705397635e-02
            6.65790429301691855101807959727012e-01 1.76545615421862284177834112597338e-01 1.67125809205001946144975022434664e-02
            1.76545615422072060818337035925651e-01 6.65790429301600039657671459281119e-01 1.67125809205244772737142255891740e-02
            2.93121007359746013620771520891140e-02 5.29565748866929753013721438037464e-01 1.71726076494239307534961369583471e-02
            5.29565748866726582200215034390567e-01 2.93121007359754374987925729101335e-02 1.71726076494305782138560800831328e-02
            1.44467382439143698924510772485519e-01 1.44467382439153968487488555183518e-01 2.50679385964746870296693259660969e-02
            3.29974011141099976551771533195279e-01 5.36181572904960379588601426803507e-01 2.85784196024791263546216413260481e-02
            5.36181572905225722891486839216668e-01 3.29974011140833856092768883172539e-01 2.85784196024844242001172744949145e-02
            5.51150751686180218058552782167681e-01 1.43779086192284799583518406507210e-01 3.05912036687950514290967873876070e-02
            1.43779086192285576739635644116788e-01 5.51150751686222073466581150569255e-01 3.05912036688056956923453810759383e-02
            3.34806658732700601621701252952334e-01 1.52961943716094977974151447597251e-01 3.48605583544493655145934951633535e-02
            1.52961943716101833601328507938888e-01 3.34806658732712258963459817096009e-01 3.48605583544514124882951477957249e-02
            3.43018349814737999459168804605724e-01 3.43018349814726564162015165493358e-01 4.36094457295439566490991012415179e-02
            ];
        
    case  13
        
        % ALG. DEG.:   13
        % PTS CARD.:   36
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.5e-16
        
        xyw=[
            2.42935351590266995769340496735822e-02 9.49305929384640645452009266591631e-01 4.15602496892741434669771294352358e-03
            2.65193427721596534929471289387948e-02 2.42695130640411434186365369214400e-02 4.17029249443872252617326878976201e-03
            9.49212602355106738194479021331063e-01 2.65067966437244201260980958068103e-02 4.17076422666404645483995139443323e-03
            3.37757637490039737404590169944640e-03 4.76731641236311620701826541335322e-01 4.39202175206313688571313491593173e-03
            4.75767229810120817479202059985255e-01 5.19892182910199562861919275746914e-01 4.61186654611970680422983193125219e-03
            5.19078319347068495304142743407283e-01 5.59127062021505007172184065211695e-03 4.94856025469582981590743742117411e-03
            8.61683974532058805095857678679749e-01 1.33996048618149239672892036878693e-02 5.08850989636669068238994384500984e-03
            1.24920975992555027822561442008009e-01 8.61305432133411819073387505341088e-01 5.17132159850462909400636846157795e-03
            1.38565453861037562316438354059756e-02 1.24773371735846919095003215716133e-01 5.20678415216160525580413320767548e-03
            2.11887064221680694597882421703616e-02 8.43843835122266905379717627511127e-01 7.94549445696764861102323607155995e-03
            8.43229678721886921621830879303161e-01 1.35456364583034505333358765710727e-01 8.01180088101775128572867856746598e-03
            1.35423179786502867472108846413903e-01 2.13482820656213274912449406883752e-02 8.01519202864208087089270549085995e-03
            3.08885351067946345438031130470335e-01 2.21919663013589105182710170538485e-02 1.07691489795688280506347211940010e-02
            6.68505759516906050698992203251692e-01 3.08901287938941060406961014450644e-01 1.09618353834781030731182482895747e-02
            2.26545012557140094944241326402334e-02 6.69170994332103252766330570011633e-01 1.09802418183186836081466353221003e-02
            2.80851540877201766299720020469977e-01 6.92471815510625865464078287914163e-01 1.19987980922641266495753953336134e-02
            6.92244674905063694936302454152610e-01 2.68723345025942798869333927314074e-02 1.20951565183311966766854794741448e-02
            2.68617447119446836611977857955935e-02 2.81009397322192155943554325858713e-01 1.21216855843665431008426836001490e-02
            1.14177848547014335056637435172888e-01 7.97358141358575411850040381978033e-01 1.39241122006051303711116062800102e-02
            7.97480792206136324118403990723891e-01 8.79806508790785812834656098857522e-02 1.40256591089016145051937556331723e-02
            8.92807293894229275776908139050647e-02 1.14502056112754088679217545632127e-01 1.41297530923219748222319225305910e-02
            1.05248789245499116407067674572318e-01 6.68690411992205024915847388911061e-01 1.72322472667379919530894483159500e-02
            6.66302228074011737568582702806452e-01 2.27505163183173791940916430576181e-01 1.79303334022174257200710201232141e-02
            2.30780373754708584277395289063861e-01 1.05457256122134482301255786751426e-01 1.81863480244095175597518476706682e-02
            1.70505915754032177922283608495491e-01 5.17406439865800038546694850083441e-01 1.97201834184251127701781314272012e-02
            5.08659397304299987219167178409407e-01 3.17052385520937729523893722216599e-01 2.02528586378061814521345240791561e-02
            3.14182386228100019032183354283916e-01 1.81070636165945320072978574899025e-01 2.06431324763724430293443390382890e-02
            4.61746081786397921664644172778935e-01 4.67859453980425554764366324889124e-01 2.10511141832600140455511450454651e-02
            6.93087496081039450279703828528000e-02 4.62285604208458600350439837711747e-01 2.10896383326258061996494319600970e-02
            4.65195525926812847306024423232884e-01 7.24357805669117588154293230218173e-02 2.12992467121988077183303289530159e-02
            2.57862585789255749357806735133636e-01 6.13139503917723982517884451226564e-01 2.25711332013103817273336204607403e-02
            6.11262776677922392565278641995974e-01 1.30036083460934759337135346868308e-01 2.28570785871127016286585842408385e-02
            1.30518213593359272417515626329987e-01 2.58171382888366274244162923423573e-01 2.29069766352318071600535631660023e-02
            4.28143799182839734918104568350827e-01 2.36200596981703009324959907644370e-01 2.56393343723977790160617473702587e-02
            3.35699578373029994260434705211082e-01 4.31102630858835600413669908448355e-01 2.58289915353369578521203209220403e-02
            2.30542429883610316965203423933417e-01 3.45601394937618655589517402404454e-01 2.58963591798315843905964328541813e-02
            ];
        
    case  14
        
        % ALG. DEG.:   14
        % PTS CARD.:   45
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.5e-13
        
        xyw=[
            0.00000000000000000000000000000000e+00 1.00000000000000000000000000000000e+00 2.65417799749999999431998576326919e-04
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 2.65417799749999999431998576326919e-04
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 2.65417799749999999431998576326919e-04
            5.73330873025999984693434896598774e-02 1.51382269814000006147303167836071e-02 3.28650590252499994262103122366625e-03
            5.73330873025999984693434896598774e-02 9.27528685716000000915926193556515e-01 3.28650590252499994262103122366625e-03
            9.27528685716000000915926193556515e-01 5.73330873025999984693434896598774e-02 3.28650590252499994262103122366625e-03
            1.51382269814000006147303167836071e-02 5.73330873025999984693434896598774e-02 3.28650590252499994262103122366625e-03
            9.27528685716000000915926193556515e-01 1.51382269814000006147303167836071e-02 3.28650590252499994262103122366625e-03
            1.51382269814000006147303167836071e-02 9.27528685716000000915926193556515e-01 3.28650590252499994262103122366625e-03
            8.15962504071100047653430920036044e-01 1.65971996956500000841572273202473e-01 6.07204817372499969230226213312562e-03
            8.15962504071100047653430920036044e-01 1.80654989724000000772541341120814e-02 6.07204817372499969230226213312562e-03
            1.65971996956500000841572273202473e-01 8.15962504071100047653430920036044e-01 6.07204817372499969230226213312562e-03
            1.80654989724000000772541341120814e-02 8.15962504071100047653430920036044e-01 6.07204817372499969230226213312562e-03
            1.65971996956500000841572273202473e-01 1.80654989724000000772541341120814e-02 6.07204817372499969230226213312562e-03
            1.80654989724000000772541341120814e-02 1.65971996956500000841572273202473e-01 6.07204817372499969230226213312562e-03
            3.16547555637800015482241633435478e-01 1.86886898773000013906919747341817e-02 7.91999665829999990873666604329628e-03
            6.64763754484899993535407247691182e-01 1.86886898773000013906919747341817e-02 7.91999665829999990873666604329628e-03
            1.86886898773000013906919747341817e-02 6.64763754484899993535407247691182e-01 7.91999665829999990873666604329628e-03
            1.86886898773000013906919747341817e-02 3.16547555637800015482241633435478e-01 7.91999665829999990873666604329628e-03
            3.16547555637800015482241633435478e-01 6.64763754484899993535407247691182e-01 7.91999665829999990873666604329628e-03
            6.64763754484899993535407247691182e-01 3.16547555637800015482241633435478e-01 7.91999665829999990873666604329628e-03
            1.92662192491999985366479819504093e-02 4.90366890375399988588611677187146e-01 8.73294867589999995738381244336779e-03
            4.90366890375399988588611677187146e-01 1.92662192491999985366479819504093e-02 8.73294867589999995738381244336779e-03
            4.90366890375399988588611677187146e-01 4.90366890375399988588611677187146e-01 8.73294867589999995738381244336779e-03
            8.75134669581000002036574869634933e-02 8.24973066083700024009317530726548e-01 9.59161334862499923015199954079435e-03
            8.75134669581000002036574869634933e-02 8.75134669581000002036574869634933e-02 9.59161334862499923015199954079435e-03
            8.24973066083700024009317530726548e-01 8.75134669581000002036574869634933e-02 9.59161334862499923015199954079435e-03
            9.35526036219000017846170180746412e-02 2.07986542316699996035822550766170e-01 1.44592372802500000739112095970995e-02
            9.35526036219000017846170180746412e-02 6.98460854061299984962829512369353e-01 1.44592372802500000739112095970995e-02
            2.07986542316699996035822550766170e-01 9.35526036219000017846170180746412e-02 1.44592372802500000739112095970995e-02
            6.98460854061299984962829512369353e-01 9.35526036219000017846170180746412e-02 1.44592372802500000739112095970995e-02
            6.98460854061299984962829512369353e-01 2.07986542316699996035822550766170e-01 1.44592372802500000739112095970995e-02
            2.07986542316699996035822550766170e-01 6.98460854061299984962829512369353e-01 1.44592372802500000739112095970995e-02
            9.74892983467000040498362523067044e-02 5.38008859514899961951073237287346e-01 1.81455421848500005055360162486977e-02
            3.64501842138300002904571783801657e-01 9.74892983467000040498362523067044e-02 1.81455421848500005055360162486977e-02
            5.38008859514899961951073237287346e-01 9.74892983467000040498362523067044e-02 1.81455421848500005055360162486977e-02
            5.38008859514899961951073237287346e-01 3.64501842138300002904571783801657e-01 1.81455421848500005055360162486977e-02
            3.64501842138300002904571783801657e-01 5.38008859514899961951073237287346e-01 1.81455421848500005055360162486977e-02
            9.74892983467000040498362523067044e-02 3.64501842138300002904571783801657e-01 1.81455421848500005055360162486977e-02
            2.21714589487299995607116898099775e-01 5.56570821025299977691247477196157e-01 2.24464131026749988406798763662664e-02
            5.56570821025299977691247477196157e-01 2.21714589487299995607116898099775e-01 2.24464131026749988406798763662664e-02
            2.21714589487299995607116898099775e-01 2.21714589487299995607116898099775e-01 2.24464131026749988406798763662664e-02
            3.86047166929600005236267179498100e-01 2.27905666140799989527465641003801e-01 2.58636133404250012635117172976607e-02
            2.27905666140799989527465641003801e-01 3.86047166929600005236267179498100e-01 2.58636133404250012635117172976607e-02
            3.86047166929600005236267179498100e-01 3.86047166929600005236267179498100e-01 2.58636133404250012635117172976607e-02
            ];
        
    case  16
        
        % ALG. DEG.:   16
        % PTS CARD.:   55
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 7.8e-13
        
        xyw=[
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 1.55064996275000002885768868310379e-04
            0.00000000000000000000000000000000e+00 1.00000000000000000000000000000000e+00 1.57879367799999994039297779302444e-04
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 1.77165038975000000819742806967838e-04
            9.39886358357699958077091650920920e-01 4.98487446340000004557646562375339e-03 1.37909290419999997802380509170916e-03
            5.43806683058000034369783293186629e-02 9.38640561861699973533745833265129e-01 1.56731019139999994300993524376509e-03
            9.39400491640000022408596436207517e-03 5.26424462697000009048231561337161e-02 1.96328522064999995075385008647118e-03
            1.64345086361999993818105991749690e-02 9.46903551735099990871447062090738e-01 2.36378709660000014061709272539247e-03
            9.46948726986200028932216810062528e-01 3.63373677167000003196406510141969e-02 2.44561278177499999073041259123329e-03
            4.26604005767999977516424792156613e-02 1.51224541798999998626351626285214e-02 2.49654108722499991704357746868936e-03
            1.22269495439000005360119516240047e-02 8.69377351066400017032265168381855e-01 3.43884547040000003795268135320384e-03
            8.67369652104699961903122584772063e-01 1.20491728577399995203656146713911e-01 3.52447945099999999929440974710815e-03
            8.45674402138899994341159072064329e-01 1.57763967870000014481757233397730e-02 3.74117160842499978448683073395387e-03
            1.39575963210299996974939062965859e-01 8.44812087037499992447919794358313e-01 3.90243759030000020651685055383950e-03
            1.31782174323099993129559948101814e-01 1.35009605583999992006738466443494e-02 3.94209233370000026314139418559535e-03
            1.57955126300000012196100840355939e-02 1.45527493853600004003467915936199e-01 4.39486365957500024065351951207958e-03
            7.36546288443600039030911830195691e-01 1.55697540907999993492483170598462e-02 5.10284600675000015596038949183821e-03
            1.39688430330000001866608272393933e-02 7.37983689445000035078692235401832e-01 5.23907196540000005130499616257111e-03
            2.54789518603899978188565000891685e-01 7.29761568977099983612788491882384e-01 5.26783532495000029333143132248551e-03
            7.31638652255499954968342990468955e-01 2.54307668331499980851617692678701e-01 5.44116900505000038873459899946283e-03
            1.57253728950999989433245929149052e-02 2.69623979579100003789449147006962e-01 5.55721021747499974163808644789242e-03
            2.66230284364699976151769078569487e-01 1.44783956308000005819280886498746e-02 5.60466734205000011170838547513995e-03
            8.67350406521400052461956420302158e-01 5.91679410399999980429974755224976e-02 5.75306542482499974622278671176900e-03
            7.41493666957000058248539176020131e-02 8.63478257506099966533952283498365e-01 5.92034756249999998684030444451309e-03
            1.59285948359999991741364766539846e-02 4.19123895523800027440586291049840e-01 6.43661608420000030950092195780599e-03
            1.56061028068000002405613102496318e-02 5.80922292114600002932434108515736e-01 6.44892004019999992570078006792755e-03
            5.91009481748399956302364444127306e-01 1.59251452650999991700331293031923e-02 6.45180819024999963051669737978955e-03
            4.03477149688900027513227541930974e-01 5.80670036810399969873230929806596e-01 6.50858080147500000361970862172711e-03
            5.69474562852600008078240989561891e-01 4.14949514630200022580197583010886e-01 6.64420354022499971835546972442899e-03
            6.78493700649999975649251382492366e-02 7.61218678590999953126328136931988e-02 6.64461904577500015595115101518786e-03
            4.26596859027200014935488070477732e-01 1.57509692312000001979122032480518e-02 6.68830823095000002315169140842954e-03
            6.70982507890000023920151761558373e-02 7.74189831242099946173595981235849e-01 9.39469516602500082091964372921211e-03
            7.52831023147999989220124916755594e-01 8.19119495638999939313151799069601e-02 9.57664735487499980959302803285027e-03
            7.75372778355699954211388558178442e-01 1.57712845729199996336333811086661e-01 9.62124237562500018994260386762107e-03
            1.68907315778699995689748902805150e-01 7.50394309974199980040054924756987e-01 9.74049564629999968523588194102558e-03
            1.68733583291899996714491294369509e-01 7.08311507268000045689504418078286e-02 9.86510278867499916521932590285360e-03
            8.21244708436000003803911795330350e-02 1.76299662677100010821362729984685e-01 1.03091194524499996099109466740629e-02
            6.28870536334499963260213917237706e-01 8.07744953317000025094785087276250e-02 1.28218109620750007199285747105932e-02
            8.11413015265999987279954552832351e-02 3.05437358977599993092866270671948e-01 1.29101410483750001639835502942333e-02
            2.96911206508000014014925227456843e-01 6.22748598887099991117111130733974e-01 1.29557510567249994232197352062030e-02
            7.67542314170999995237565372008248e-02 6.24724714954600024441333516733721e-01 1.32131997045249992284166040690252e-02
            6.22302233384499969837122534954688e-01 3.01148582116600005420536945166532e-01 1.34626393256749993243559515576635e-02
            3.10378628805099976428749641854665e-01 7.79098365078999960386951784130360e-02 1.35473832329750000502688322967515e-02
            8.19218215187000053933985554976971e-02 4.60363303835099990646284595641191e-01 1.46184286611000004152405296053985e-02
            4.71702266501299982692785306426231e-01 8.21554006796999941597547945093538e-02 1.48215792090750000248311479822405e-02
            4.54660341525000011309032288409071e-01 4.63756503388999985482143983972492e-01 1.48589569187250001430777857081011e-02
            1.70109133923700001078671562027012e-01 6.42227780818800053630468482879223e-01 1.57950063965749988581510621088455e-02
            6.40600432948699949342596937640337e-01 1.89829353725600002977813574034371e-01 1.58231711288249997371391231126836e-02
            1.91226758371700000793680374044925e-01 1.73995568534300010243853762403887e-01 1.60176840442999984803762458795973e-02
            1.88531576706999992598667859056150e-01 4.79891407040600015410092282763799e-01 2.03010148979500017196730965451934e-02
            4.77292995769100003577989355108002e-01 3.34835659811899999382944770331960e-01 2.03609378382499997539589742245880e-02
            3.12697462175999973954532151765306e-01 4.95797219725900017994746349359048e-01 2.03669800310249983488208869175651e-02
            4.96122594594600008210960595533834e-01 1.92755366890399987278570392845722e-01 2.03762637020999982928604055132382e-02
            1.92880531286699991033728451839124e-01 3.16101580726100006568657363459351e-01 2.03791166234749999974784628875568e-02
            3.36004145381600005659805674440577e-01 1.89489280128999987296012363913178e-01 2.04232764905750013850749269295193e-02
            3.33728055084800023788460521245725e-01 3.34357102181099985482859437979641e-01 2.30804583632749985011578530702536e-02
            ];
        
    case  18
        
        % ALG. DEG.:   18
        % PTS CARD.:   66
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.8e-16
        
        xyw=[
            1.16731059668422967878775864392082e-02 9.81256595128901731861503776599420e-01 6.29143924661216820230780211176125e-04
            9.81003085838796140549789015494753e-01 7.14625048631965624490458566242523e-03 6.31836300180475189496032140112902e-04
            1.06966317091656323151616092559379e-02 1.15153933375962483354193466311699e-02 8.31732383329287648827954360086778e-04
            9.38247698355051062968357200588798e-01 4.95570591340603744390236329309118e-02 2.03758730313529333455502268179771e-03
            1.26627518417216609891706724511096e-02 9.37012362061508063959536229958758e-01 2.15338814354033630457663228696674e-03
            5.98109409983654249920448364719050e-02 1.21364578921858024679325538386365e-02 2.19466865448246202865978382590129e-03
            1.37363297926714009872251054389380e-02 6.12783625596970207838332100891421e-02 2.42748963904181871628629174608704e-03
            9.22952795940536918628538387565641e-01 1.41128270602437974612364257609443e-02 2.56165529787218213680910494645104e-03
            6.33107354992725884779147804692911e-02 9.22019729172741109302080531051615e-01 2.70994220851850271813976434032156e-03
            1.17265100334620151617537331389940e-02 1.50052047522906850662849365107832e-01 3.23463475439669930974750933216910e-03
            1.55472058732349788234827769883850e-01 8.32514712158920699991426772612613e-01 3.40849558957418731497401864771746e-03
            8.34329388898202961577510450297268e-01 1.25228158758818042262106473572203e-02 3.46193320366632029116971125404234e-03
            8.50163803195680856106264400295913e-01 1.37199750873573111942249624917167e-01 3.48553850262133674275943917564291e-03
            1.28816350521985811816705336241284e-02 8.47762706347928829408999717998086e-01 3.60303499919003081264357035706780e-03
            1.51080160895852660951632628894004e-01 1.36526924039371260616304581958502e-02 3.84258638835027971705771143717811e-03
            1.01917879216608664449950083508156e-02 5.77043861834499960394850859302096e-01 4.06224505631521976695141873392458e-03
            2.81337239930328442127915877790656e-01 7.06685375962319062281835613248404e-01 4.24295760700375574475673445817847e-03
            7.12437462850067815267607329587918e-01 1.24569780989904812379398535426844e-02 4.25221331053355234425028541522806e-03
            2.76302525086301431489488322768011e-01 1.21741311385635056496745676213322e-02 4.27383801686573045613970478484589e-03
            1.09658368560642989486542120403101e-02 4.19430671246641517679876187685295e-01 4.34722136397843513661021219718350e-03
            4.28911051788389785510702267856686e-01 5.59961606746893436614698202902218e-01 4.36359906096797558333610922431944e-03
            4.21542055511432756098599838878727e-01 1.16475994784655950686769187996106e-02 4.46016893216582156062033348575824e-03
            5.71125859044400052688672531076008e-01 1.18218313988502476269415453202782e-02 4.46117159698410439827576112747920e-03
            5.82686827051093314544516488240333e-01 4.05788958117719755769314815552207e-01 4.47615843880835515000704560861777e-03
            1.30567806713243420718706389038744e-02 2.72502375086800474601744781466550e-01 4.53149390501767553723899695228283e-03
            1.30760400963916122857311208349529e-02 7.22471252323362400638018243625993e-01 4.61962097205037124786342062066069e-03
            7.26343706240680786478947084106039e-01 2.60298401925061928530169552686857e-01 4.64483910927785282468649441511843e-03
            6.87230068637360302918892784873606e-02 6.31417277209678357019484451484459e-02 5.08042879441409581803279493783521e-03
            8.65230210152928114197834474907722e-01 7.20611837337742783260452483773406e-02 5.34429154523023784012503156759522e-03
            6.48599071037404689699101822952798e-02 8.59043354390996882763431585772196e-01 5.79792135245708016577648180600590e-03
            1.48349494336221521262331179968896e-01 7.88878835223944308019383697683224e-01 6.86066777147933265501578148359840e-03
            6.24359898395992346520344540294900e-02 1.49393549935431940722807553356688e-01 7.25754805850912284442832955733138e-03
            7.87136901173496439376719990832498e-01 6.56382042756523814697189322941995e-02 7.36306846263517882078541987311837e-03
            5.19104921609673947147101102927991e-02 5.25563569560499987964874435419915e-01 7.48590629072923632869596133332379e-03
            1.54312992744386756616137290620827e-01 7.16383926916974306253749205097847e-02 7.67567370296964261028316300894403e-03
            2.61784274560300012080205078746076e-01 6.21479485288090535699545569059410e-02 8.13158414656712548451711342067938e-03
            7.66725787281274606677072824822972e-01 1.65821155483138971131751304710633e-01 8.19710521265225137976795366512306e-03
            2.58210367662758732976158171368297e-01 6.80011976613870805508099692815449e-01 8.28086687980062081948506857997927e-03
            6.79065925147462334798831307125511e-02 7.57151543778185054378582208300941e-01 8.65418817186469797875236764639340e-03
            5.29357827480446907486566487932578e-01 4.12150384110698542183826020846027e-01 8.67703434940207164738357192845797e-03
            6.66036150484166367125027363726986e-02 2.61251308788644831526681855393690e-01 8.68430123509580237373128852595983e-03
            5.85675461899543767230191804173955e-02 3.90223611453477114974219830401125e-01 8.71321906135587818942322968496228e-03
            6.44535360410797841801411323103821e-02 6.37362655976104197641518567252206e-01 8.71503902964948506948417161765974e-03
            6.74813842915122696908269972482231e-01 6.37583342061289748414765199413523e-02 8.88678924937164917141618758478216e-03
            3.91460231036899974910170385555830e-01 5.50323809056282087226463772822171e-01 9.00457490956721104402582511738728e-03
            6.48770149230723958133637552236905e-01 2.83672836026290065003507834262564e-01 9.07315714606773439376841139392127e-03
            3.94649822040804787448564638907555e-01 6.05175522553690475868748421817145e-02 9.54744255207919632899571382722570e-03
            5.39013715193328413199935766897397e-01 6.11990176936391058748476723394560e-02 9.80632000294656691607997345272452e-03
            1.62789508278483124881219623603101e-01 6.86132214103464810328603107336676e-01 1.20677531471879689178861028153733e-02
            6.81243632264075471560715868690750e-01 1.56796834589900596634848284338659e-01 1.22478030391557947409442164143911e-02
            1.54283287802026880530803509827820e-01 1.66751262401978767035970463439298e-01 1.24305208468046303471599145495929e-02
            2.52272775044500019436810589468223e-01 2.50480393339460316592948174729827e-01 1.26766434246480385467092943940770e-02
            2.54798153240699987165385209664237e-01 4.99409064904299981879631786796381e-01 1.27442998510756983626679428311945e-02
            1.48558054919430054807705232633452e-01 5.75602309608733597734442355431383e-01 1.30340015916796114758824387536151e-02
            2.93023960643626679711815086193383e-01 5.65689735416167849280100199393928e-01 1.30865218731192906737037517928002e-02
            2.80899127230993306270079301611986e-01 1.43792157424766253726033937709872e-01 1.31110170887910690989963669039753e-02
            4.82098959297124984857418894534931e-01 2.51855753586498587459630016383016e-01 1.31864911205620913975389285610618e-02
            5.64187824544361338219289336848306e-01 1.46296674315256225451520322167198e-01 1.32362265931905524751766023427990e-02
            1.30769964434413243914434588077711e-01 4.48957758611656376768195286786067e-01 1.35598898625217834795764204613988e-02
            1.47969222194773153411517796484986e-01 3.00117438682896364721131021724432e-01 1.35867550854841621227331316390519e-02
            5.63868422294613202971902410354232e-01 2.81377208929745936671906747506000e-01 1.36775137159699782668509371319487e-02
            4.36115742879063184034293954027817e-01 4.25205344642040183877895742625697e-01 1.39322086478155908462195355923541e-02
            3.60326393528606292449012471479364e-01 2.59919000488856355435274281262537e-01 1.44433566058263379572323970023717e-02
            4.22418833467427257133408602385316e-01 1.45323844330263851620799187003286e-01 1.46348445405671198638053454033070e-02
            3.71900183305260034760664211717085e-01 3.78012270356698043549670273932861e-01 1.52259812669901951204254331173615e-02
            2.41364500692875016962801737463451e-01 3.84756328493939148138736072723987e-01 1.59318491112365318018717630366154e-02
            ];
        
    case  20
        
        % ALG. DEG.:   20
        % PTS CARD.:   78
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.0e-16
        
        xyw=[
            8.94113371120544217496295402725082e-03 8.69832937019905140518805097826771e-03 5.43613634964998197751651076714552e-04
            9.79262262980672426593287127616350e-01 1.02644133743667476804661831124577e-02 7.24678381627871098791260351390520e-04
            1.05475382111858816819305317835642e-02 9.78551420251513315307079210469965e-01 7.71150733413381744320302946249512e-04
            2.37770619471529233088236132687143e-03 6.36551098603544696086231624576612e-02 8.60040827590111633081659814337172e-04
            6.30425115794740753560887469575391e-02 4.15063475086186860901760198316879e-03 1.04746180029398490950176547897854e-03
            9.30842249673007549048975306504872e-01 4.80534822625924146199949404945073e-03 1.11845128744862297096029912779613e-03
            6.29076555490187122599010649537377e-02 9.31679006948127330645093024941161e-01 1.17636052035514438243724022470360e-03
            9.31596224638061132239386097353417e-01 6.26264881801440742581377207898186e-02 1.22169839375744831641568133306919e-03
            6.19516894144997096183224982723914e-03 9.29358705856405231671146793814842e-01 1.29819108422282530022895397081584e-03
            2.87125819236766499642499184119515e-02 3.10202122997135891491193859792475e-02 1.85182647452015666789115755364037e-03
            9.29384447830515436628218139958335e-01 3.42152968218708067316136123281467e-02 1.99388525752322879822098222746263e-03
            3.75457566621083954627735579379078e-02 9.25786888466940571795760206441628e-01 2.08876307273982804099676791054208e-03
            8.68957390638382641545600648669279e-03 1.58497125150980244789877815492218e-01 2.40416652159353800774499454462330e-03
            1.54759705396456032078233988613647e-01 8.36360665768839717237881359324092e-01 2.40795644626126984985670098637911e-03
            8.33102529418505488578716722258832e-01 8.92572448244856840593453028986914e-03 2.46443651895836930362793104620778e-03
            8.37423107352594398022915811452549e-01 1.52916730407855333551125909252733e-01 2.56644700753416461075362420274359e-03
            1.55936250523392189171190125307476e-01 9.49662400580270418815054256356234e-03 2.57970257777746445060351732081472e-03
            9.85996420952243816937521359022867e-03 8.34221149359543723278420657152310e-01 2.65727504076160954069041331138123e-03
            4.05587373328940736794123722575023e-01 7.43893020079155944251914078790833e-03 2.67203267238412794232216285195136e-03
            5.96472789861799967958688739599893e-01 3.95633080931094838206263375468552e-01 2.67422552525793709790602470377507e-03
            8.07478004157673352747526251960153e-03 4.03131942590241654222893430414842e-01 2.72566154284044608741011117558628e-03
            7.50739777207003189385403629785287e-03 5.85160959468056574017680304677924e-01 2.74749458936237016953851863831915e-03
            3.93676451923723857984072083127103e-01 5.97489659289885510773387977678794e-01 2.83557638072564786521634339067077e-03
            5.84653072621227565264234726782888e-01 8.72504649681960300910965599996416e-03 3.01339107324613770036436655175294e-03
            4.87080411211974706464644668812980e-01 2.02129229911995403345947863726906e-02 3.49047984551459616192614632268487e-03
            2.68351281178442424568686419661390e-01 7.20234008866831643125294704077533e-01 3.52869978841008000158829815973149e-03
            7.22395628874778816808088777179364e-01 2.66239936645638941570268798386678e-01 3.54825867614600451785400458959430e-03
            2.71682674235739995971528060181299e-01 1.12882698808235912291131342044537e-02 3.60531690670821882566121630020461e-03
            1.12580842045890989339129717450305e-02 7.16969596332504899294235656270757e-01 3.61760867137253874781244178393536e-03
            1.15034734369742191323338076358596e-02 2.74006711016551274262553761218442e-01 3.62374424680139304785764231553458e-03
            7.14052590056392100237303566245828e-01 1.13511560497063217906044130245391e-02 3.63466939235189802695491501083325e-03
            4.90287105311137949570365890394896e-01 4.93649184146848774012994454096770e-01 3.64910477313937827575918149136669e-03
            2.01423425209245159084314735764565e-02 4.83257345960130979545255058837938e-01 3.68286446165711845238122101875433e-03
            3.61107464858592894230149283885112e-02 9.35679501581969663703830519807525e-02 4.18659908260370777766112837525725e-03
            8.60799881985092274483406526996987e-01 3.97379067075453867863288337503036e-02 4.22388751143842194890609675894666e-03
            1.00589152600109088542268409582903e-01 8.58634341935195832817839800554793e-01 4.23556657210420652720506140553880e-03
            9.18740717058512956771920698884060e-02 3.95513001973352915086046266424091e-02 4.32675430238721459019757276109885e-03
            8.60488829619093453793254866468487e-01 9.66224057079418385818314618518343e-02 4.36311366231690103689722093349701e-03
            4.39842178673168632263568156304245e-02 8.56188634910677426326230943232076e-01 4.43043055398302000613686146834880e-03
            2.01101760673522977906912956314045e-01 7.44911583562643420108884129149374e-01 7.07060060058183146441956878902602e-03
            7.44999372626340083947127368446672e-01 5.36865638165824748950782918655023e-02 7.12491781220321931944461368857446e-03
            5.32186641309854713277616156119620e-02 1.96375427593497742240202796892845e-01 7.12514116346794294104549649659930e-03
            7.45398464740034905062771031225566e-01 1.98206580555018330169758655756596e-01 7.51618058694255870150202625268321e-03
            1.95728993287614144724884113202279e-01 5.55713833156083122921664596560731e-02 7.55078192705701829040565442596744e-03
            1.09253205798743646681714380974881e-01 6.10003618241300005209382106841076e-01 7.59967840192289374040557348166658e-03
            5.67625702000490173215752065516426e-02 7.40912189495913087178280420630472e-01 7.64171990184990632816175448738250e-03
            4.83837933474768472352778303502419e-02 6.07513566097787838238275526236976e-01 7.65168532505686345424056327146900e-03
            1.08061280976018470045829644732294e-01 1.12208151043696469084665068294271e-01 7.73325170501491885449718566292177e-03
            6.18560590099037632683121046284214e-01 2.69875370303500006219366014192929e-01 7.74434552087149147281408545495651e-03
            7.72129601349661243148148059844971e-01 1.11411739533305548244257465739793e-01 7.82865626361630741192332294531298e-03
            6.11573480113248968415007311705267e-01 3.38936767793083126854014608397847e-01 7.83933733479477007355384898801276e-03
            3.38132610337598005223469499469502e-01 4.94693938787471032547671256907051e-02 7.85801173216685126809988304330545e-03
            1.17308412825418845182312566066685e-01 7.69645130979520919467518069723155e-01 7.87955359733825272394724947844225e-03
            2.67455126059625236667471881446545e-01 1.11571880815410667819875811801467e-01 8.10620344963784537906104077364944e-03
            6.54210016002574556992499310581479e-01 1.90654831469986796310678300869768e-01 8.68780380964838047641585205838055e-03
            5.38297481157758231362642220574344e-02 3.35861682684902740358978689982905e-01 8.75983637317068186689539288636297e-03
            1.84884032411679721796815556444926e-01 1.55183152385134082162920776681858e-01 8.76793550775805188846057092177944e-03
            3.37626710474410451912063990675961e-01 6.08140259629467938573554874892579e-01 8.80323038335373113383308663060234e-03
            6.06710203449961738009221789980074e-01 5.42632795598224151878419263539399e-02 8.81538762453739146995879139012686e-03
            4.61261408549577645032968575833365e-01 6.88176670721690475485132765243179e-02 9.16008050856753283652800234904134e-03
            1.52546536567121610028507916467788e-01 6.51024084574876371611651393322973e-01 9.19332769175816577611470847841701e-03
            7.00582543543042252087715837660653e-02 4.66190439274151935400425372790778e-01 9.29189157342165394926070121073280e-03
            4.70420137903155399783372558886185e-01 4.63482645535340076481389814944123e-01 9.33428929015709361871611093874890e-03
            1.21646169374596019308043537421327e-01 2.38149487551563759923922702910204e-01 1.00993336646966504199784964157516e-02
            6.37140405270220533395786333130673e-01 1.23839938451338588132344398218265e-01 1.03395010159411937511464429917396e-02
            2.37990451511855244381976604017837e-01 6.37021645232653477286532961443299e-01 1.05489447967511071335700023610116e-02
            1.48392985717712366788845201881486e-01 4.89418857778011140435836523465696e-01 1.23862751009288403480912421628091e-02
            3.59806957154975348700531867507380e-01 1.45288086625325219003457277722191e-01 1.25104815285276768643818456894223e-02
            4.94144105509497388872830470063491e-01 3.61021638381830600739164083279320e-01 1.26448646778648133431888211930527e-02
            1.44063068798082505317381674103672e-01 3.51350834188701666160881131872884e-01 1.30009302546985807541579660551179e-02
            5.01976444000366672781865418073721e-01 1.43549166329309002554381891059165e-01 1.30383391971379321988555588518466e-02
            3.55542383429807651307186233680113e-01 5.01649159950201317670348544197623e-01 1.31224788089477160252238618909360e-02
            2.44343954077136132907455134954944e-01 2.40605212910407678661428576560866e-01 1.49789940628951411677460825444541e-02
            2.43706498934185250559991686714056e-01 5.10901727705534658419139759644167e-01 1.49902499356569345045908647762190e-02
            5.12220080732083804520016201422550e-01 2.45273797354294331718449484469602e-01 1.49978818032193463027912372353967e-02
            2.52603831517775656578805865137838e-01 3.70031955509367738166304206970381e-01 1.58533295862338351189801954888026e-02
            3.75989565285069105016191315371543e-01 2.50540661163056177507257871184265e-01 1.58827965277093911822259286736880e-02
            3.72907798714407878737375767741469e-01 3.75375027754923540346254640098778e-01 1.59301651418000574100819477507684e-02
            ];
        
    case  21
        
        % ALG. DEG.:   21
        % PTS CARD.:   91
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.0e-16
        
        xyw=[
            3.55243919220995813673669161403268e-03 3.55243919219624861710604690756554e-03 1.67610910967364371004784118213138e-04
            3.55243919220429599931110331567652e-03 9.92895121615643416745911054022145e-01 1.67610910966581116250326277672400e-04
            9.92895121615600562137160522979684e-01 3.55243919219402426792897564666873e-03 1.67610910967279450869624191078344e-04
            9.55354827372969173104877427249448e-01 8.78989290927685053278484161864981e-03 1.13681520185093961304578780158181e-03
            3.58552797177463869471303326008638e-02 8.78989290927979956269400219071031e-03 1.13681520185086913990457624379360e-03
            9.55354827372995485390561043459456e-01 3.58552797177243073867280998001661e-02 1.13681520185094893718447117691994e-03
            8.78989290927446355328189753208790e-03 3.58552797177304483078330576972803e-02 1.13681520185046798510075660715302e-03
            8.78989290927445314494104167124533e-03 9.55354827373050552452582451223861e-01 1.13681520185051850892199443165964e-03
            3.58552797177701526587512148580572e-02 9.55354827372945747399057836446445e-01 1.13681520185207260431603515371535e-03
            8.86526487904757964919610913057113e-01 1.08232974501700593417119478090171e-01 1.30193963299627779410527761427829e-03
            8.86526487904712667820206206670264e-01 5.24053759349522689514611073491324e-03 1.30193963298891302558801807975897e-03
            5.24053759350145108297791551876799e-03 1.08232974501748444029480822337064e-01 1.30193963298957915940279317368322e-03
            5.24053759352011410549421199789322e-03 8.86526487904805704509669794788351e-01 1.30193963299251496204544942258963e-03
            1.08232974501802803324324031564174e-01 8.86526487904688242913664453226374e-01 1.30193963299116794926635343188082e-03
            1.08232974501755729868079924926860e-01 5.24053759349578460874363727839409e-03 1.30193963298870767769654932521917e-03
            4.66397432149654941890837278606341e-02 9.06720513569923891594726228504442e-01 1.63588582216670310777451025074924e-03
            4.66397432150000013084678585073561e-02 4.66397432150258764438355285619764e-02 1.63588582216900443530582798246087e-03
            9.06720513570000052894215514243115e-01 4.66397432150000013084678585073561e-02 1.63588582216310420708316186733100e-03
            2.07572045694564971674012099356332e-01 8.27592412838175708889032478055015e-03 2.31844603832076371144332860296799e-03
            2.07572045694622564493414529351867e-01 7.84152030176993752874636811611708e-01 2.31844603832126764861309986542892e-03
            7.84152030177069025995706397225149e-01 2.07572045694540824323226502201578e-01 2.31844603832395169951130498020575e-03
            8.27592412838997620871950289256347e-03 7.84152030177113656961296328518074e-01 2.31844603832362080100826240425249e-03
            8.27592412838307027456163922352061e-03 2.07572045694563389606202008508262e-01 2.31844603832122991837749736987462e-03
            7.84152030176999970123574712488335e-01 8.27592412837836050032436219225929e-03 2.31844603832017694122757944796831e-03
            8.58119489724999962021456667571329e-02 3.14836947700999972288293804467685e-02 2.39844456558702514084591150833603e-03
            8.82704356257399957996767625445500e-01 3.14836947700999972288293804467685e-02 2.39844456558927724559859839814635e-03
            3.14836947700999972288293804467685e-02 8.58119489724999962021456667571329e-02 2.39844456558572670032414286822586e-03
            8.58119489724622208637327958058449e-02 8.82704356257399957996767625445500e-01 2.39844456559133159187502393194791e-03
            8.82704356257399957996767625445500e-01 8.58119489723861289531825491394557e-02 2.39844456558559876446778957870265e-03
            3.14836947701365027496578363752633e-02 8.82704356257399957996767625445500e-01 2.39844456558263498940908320378185e-03
            6.68877823382516334405067937041167e-01 9.51507606247778067731779572113737e-03 2.85619522917313723928134372442855e-03
            9.51507606247835140134139209067143e-03 3.21607100554938851733055571457953e-01 2.85619522917295943012505610170138e-03
            9.51507606248120849090632589195593e-03 6.68877823382600045221124673844315e-01 2.85619522917538804299142363163355e-03
            6.68877823382546310426732816267759e-01 3.21607100554971603312282013575896e-01 2.85619522917574539602747485389500e-03
            3.21607100554981262252596252437797e-01 6.68877823382539427043980140297208e-01 2.85619522917256391317253338968385e-03
            3.21607100554938130088089565106202e-01 9.51507606247852834313594172499506e-03 2.85619522917306828402317364634655e-03
            4.37999954311300010445506813994143e-01 9.98597856805449532457963357501285e-03 2.93042410433808553013590447733350e-03
            9.98597856805516492784136062255129e-03 5.52014067120596196858173243526835e-01 2.93042410433984237133619998871836e-03
            4.37999954311300010445506813994143e-01 5.52014067120644713604349362867652e-01 2.93042410433717870343883760142489e-03
            9.98597856805430623972075210303956e-03 4.37999954311300010445506813994143e-01 2.93042410433826941082435801888550e-03
            5.52014067120551343847978387202602e-01 4.37999954311393047134970402112231e-01 2.93042410433925343271610586270981e-03
            5.52014067120565998791903439268935e-01 9.98597856805341979602452795461431e-03 2.93042410433803609051683913833131e-03
            7.97493107214751639766348034754628e-01 4.05093994118307862417616149741662e-02 4.70492888080049301985363641165350e-03
            4.05093994118551278815765215313149e-02 1.61997493373374240599460449629987e-01 4.70492888080182615484492458790555e-03
            4.05093994118959702110949194775458e-02 7.97493107214784724412481864419533e-01 4.70492888080456615057522995471118e-03
            1.61997493373414375161800649038923e-01 7.97493107214728103038225981435971e-01 4.70492888080357302138523323264963e-03
            7.97493107214813035099609805911314e-01 1.61997493373290002427467015877482e-01 4.70492888080464161104643494581978e-03
            1.61997493373358253387905847375805e-01 4.05093994118448305630231232044025e-02 4.70492888080021025992705219209711e-03
            3.86421555195467947285692389414180e-01 3.86421555195499977219952825180371e-01 5.88152450676362407305797219692067e-03
            3.86421555195499977219952825180371e-01 2.27156889609037238031419292383362e-01 5.88152450675447253936045655109410e-03
            2.27156889608964795979062500919099e-01 3.86421555195499977219952825180371e-01 5.88152450677140777729468013035330e-03
            8.09012937932865439094598514202517e-01 9.54935310335075365761881016624102e-02 5.88928665376745562320959237467832e-03
            9.54935310336080256377044861437753e-02 8.09012937932828912757088346552337e-01 5.88928665376993280833328725520914e-03
            9.54935310335289361249877515547269e-02 9.54935310335349868404719586578722e-02 5.88928665376761781985459620614165e-03
            2.74542523871785681954804658744251e-01 4.79840480721273610065225057041971e-02 6.70615518575797681100070235515886e-03
            4.79840480721468107261351576653396e-02 6.77473428056089232995873317122459e-01 6.70615518576100390346628188353861e-03
            6.77473428056023729837420432886574e-01 4.79840480721209425296613915179478e-02 6.70615518575759170238903550398391e-03
            6.77473428056128090801735197601374e-01 2.74542523871725785422626131548896e-01 6.70615518576055287536252791369407e-03
            2.74542523871874555307925902525312e-01 6.77473428055990978258193990768632e-01 6.70615518576009230627965607141050e-03
            4.79840480721299630917364709148387e-02 2.74542523871823651582246839097934e-01 6.70615518575945999957266252522459e-03
            4.05347244666806272306303071673028e-01 5.42984962234297929839499374793377e-01 7.85724441946966840888411098831057e-03
            5.16677930988909961396160497315577e-02 4.05347244666757422493219564785250e-01 7.85724441946935615865843516303357e-03
            4.05347244666699413340182900356012e-01 5.16677930988905728670879113906267e-02 7.85724441946800827851760118392122e-03
            5.42984962234337564801478492881870e-01 5.16677930988870409700908226113825e-02 7.85724441946785909229866717851110e-03
            5.16677930988952704982608565842384e-02 5.42984962234409396231171740510035e-01 7.85724441946977596173962155035042e-03
            5.42984962234452250839922271552496e-01 4.05347244666650619038250624726061e-01 7.85724441946972912420577017655887e-03
            1.87773861553900633936109443311580e-01 1.06814826758807768225523204819183e-01 8.42990480396907208848755743701986e-03
            7.05411311687231146194676512095612e-01 1.87773861553885312858369616151322e-01 8.42990480397139488322189038171928e-03
            7.05411311687196063147098357148934e-01 1.06814826758797304373516112718789e-01 8.42990480396925249972905902495768e-03
            1.06814826758885511592822581405926e-01 7.05411311687161868277939902327489e-01 8.42990480397190489192382756300503e-03
            1.87773861554004939389272976768552e-01 7.05411311687162645434057139937067e-01 8.42990480397134284151761107750644e-03
            1.06814826758819994556581889355584e-01 1.87773861553926002532222128138528e-01 8.42990480397054833816561369985720e-03
            1.19505971200887586847194654637860e-01 3.05712299064265724535260915217805e-01 1.06936323553343173159424495111125e-02
            1.19505971200916022434412866459752e-01 5.74781729734796575925770412140992e-01 1.06936323553354015181149350155465e-02
            5.74781729734786916985456173279090e-01 1.19505971200873140070086719788378e-01 1.06936323553323709562024035335526e-02
            5.74781729734875623805123723286670e-01 3.05712299064210824006693201226881e-01 1.06936323553346382397855052204250e-02
            3.05712299064345549570731463973061e-01 5.74781729734756718919186369021190e-01 1.06936323553344682368848594933297e-02
            3.05712299064230197398472910208511e-01 1.19505971200881550009498255349172e-01 1.06936323553325392243795732838407e-02
            5.98124574336285963482851002481766e-01 2.00937712831805004309870810175198e-01 1.10284733184225387792043449053381e-02
            2.00937712831901760246466892567696e-01 5.98124574336253878037439335457748e-01 1.10284733184214580464788113545183e-02
            2.00937712831833842352935448616336e-01 2.00937712831819270675737243436743e-01 1.10284733184219697899042245126111e-02
            2.16077520000508976893982548972417e-01 3.12136025667285765816671982975095e-01 1.15367398671022431538846220178129e-02
            3.12136025667357375201760305571952e-01 2.16077520000487938167665902255976e-01 1.15367398671085870376362692013572e-02
            2.16077520000537232069959259206371e-01 4.71786454332146831802674569189548e-01 1.15367398671020228440031729633120e-02
            3.12136025667380412329521277570166e-01 4.71786454332108307063720076257596e-01 1.15367398671000556675814152640669e-02
            4.71786454332162707991926708928077e-01 3.12136025667294259022810365422629e-01 1.15367398671083927486069597989626e-02
            4.71786454332112858978121039399412e-01 2.16077520000476058781302413080994e-01 1.15367398671030272488957635346196e-02
            4.37657990384939132866293221013620e-01 4.37657990384801298677786007829127e-01 1.17288117156125019924228780610065e-02
            4.37657990384825723584327761273016e-01 1.24684019230242804088248931293492e-01 1.17288117156112599304140786671269e-02
            1.24684019230249965026757763553178e-01 4.37657990384861306232266997540137e-01 1.17288117156118393280550549206964e-02
            3.33333333333324155489663098705932e-01 3.33333333333314052460139009781415e-01 1.37799995086866841620398815848603e-02
            ];
        
    case  23
        
        % ALG. DEG.:   23
        % PTS CARD.:  105
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 8.7e-16
        
        xyw=[
            8.78093038360354405380547859749640e-03 9.90367643677188547179923716612393e-01 1.60957456520614291097318049317266e-04
            9.90367531421999958141100250941236e-01 8.78092162321842534256077783538785e-03 1.60960326908807319903557675466743e-04
            2.70292764499421205373619336853608e-03 3.35914404438093713856083866176050e-02 2.53368392755180934713954421511062e-04
            3.35909214524000021562244455708424e-02 2.70289467097205998724618325468327e-03 2.53368814399491221710236654729442e-04
            9.16750686061702790841110299879801e-03 9.16763530514256783088278268678550e-03 4.91998248372046563699211407794110e-04
            9.67556818255816075691200239816681e-01 8.47371766560996164674435249253293e-03 8.36682844597088173389798360091163e-04
            8.47372006880422676267095027924370e-03 9.67556943534542246432295087288367e-01 8.36683480207325684652186481571334e-04
            7.87819487922387061418216802621828e-03 6.76784943861999943592877571063582e-02 1.07183308438260778937267669874700e-03
            6.76785477699999998835167502875265e-02 7.87816592909016365220420397008638e-03 1.07183649713801397093670964011380e-03
            9.47026695504659787872014931053855e-01 4.42974541187251200091701264227595e-02 1.07509503430154259619178080953361e-03
            4.42974755679406348440885210493434e-02 9.47026667648678599498168750869809e-01 1.07509622745514000358157957748517e-03
            9.14424321488164104643203700106824e-01 8.17354551315039798420336580875301e-03 1.42336573012743572777538680895759e-03
            8.17354244589089257155833934120892e-03 9.14424423403110364638735063635977e-01 1.42336600333980794162724858864522e-03
            2.49745229274100005723013850911229e-01 3.83323243471999974829600432713050e-01 1.54109670035983592657757856869694e-03
            3.83323264605499980373082280493691e-01 2.49745126800500011343686423970212e-01 1.54111891046524395430805665085927e-03
            8.87685035355727181105578438291559e-01 1.03532880944553346003189631119312e-01 1.55036283976982876969819002965778e-03
            1.03532922829663281971690480531834e-01 8.87684993183984527576058098929934e-01 1.55036329879286926755610487305148e-03
            7.72559236181191190395711743121865e-03 1.40319099197360630526532077055890e-01 1.74090825734851792523649738342328e-03
            1.40319242510639513588088789219910e-01 7.72559346238606541729110332994424e-03 1.74090829604721972038994248777044e-03
            8.10459100965200018862333308788948e-01 1.80964252392600916108378328317485e-01 1.87665644300745990992462797208873e-03
            1.80964300371699998715158130835334e-01 8.10459051533444885251356026856229e-01 1.87665661412736114489752115019883e-03
            8.33076794868340519961691370554036e-01 8.30109396770613121308013404586745e-03 1.97686920847301534839846048896561e-03
            8.30109071261880826497758789628278e-03 8.33076854539217759310076871770434e-01 1.97686931212276819150308604378097e-03
            3.48407706147432008614295284587570e-02 3.48406969481684300471435733470571e-02 2.00883361556515723997384803567456e-03
            2.74028767960759755961674954960472e-01 7.17398184794867055380507281370228e-01 2.19908602683966747717847489695941e-03
            7.17398222477858449330767598439706e-01 2.74028730438602985408635959174717e-01 2.19908620281223862735764384979120e-03
            2.39497685823422484796552112129575e-01 8.18591822619156410378860044829707e-03 2.28260489288794731627119105610291e-03
            8.18591858449671701258854028537826e-03 2.39497556667719574408081939509430e-01 2.28260534028300494935326270251608e-03
            6.88361520745116479969016864970399e-03 4.84374089268735374957941530738026e-01 2.32054371876321821263622346975808e-03
            4.84374148569916229689624742604792e-01 6.88362329487813943973417707411500e-03 2.32054539155624210342021562780701e-03
            4.96076777274090274794104971078923e-01 4.96076752950683486087513074380695e-01 2.36249515443846033052599509005631e-03
            6.11293677624577647122805501567200e-01 3.80432369123821667589169237544411e-01 2.36568671209847619651966965648171e-03
            3.80432398034445951040538602683228e-01 6.11293646653274347357864826335572e-01 2.36568713235994895413560890062854e-03
            7.30389071352401275660781720944215e-01 8.39871797008916350091300984104237e-03 2.38889430712179882607126835125655e-03
            8.39871686392917003727287550418623e-03 7.30389089540734115502118584117852e-01 2.38889482107646150890745140316085e-03
            6.12852567561200767265461308852537e-01 7.54759796947155042062815866188430e-03 2.40347106219925389994362951995299e-03
            7.54759610367729868313224628195712e-03 6.12852548458182599944166213390417e-01 2.40347117064719202964018229806697e-03
            7.95253165133135049569279573233871e-03 3.55977382672119313511416294204537e-01 2.49978810530322422336912957518962e-03
            3.55977487045950280464268189462018e-01 7.95253585022433380491424514957544e-03 2.49978879626121079648992662214368e-03
            9.11023697796606413135123148094863e-01 4.37233665344507055339207113320299e-02 2.50753298191521431861894519954603e-03
            4.37233605166329891700272014531947e-02 9.11023680744558528310506062553031e-01 2.50753366588934208797345704056170e-03
            3.88480061834737405712658642187307e-02 9.67030908281891893318871211704391e-02 3.12341690463335645844278332106114e-03
            9.67032117936429402860198933922220e-02 3.88479942386262128151130923470191e-02 3.12341815311715153766214392305756e-03
            8.73226911311275888483152129992959e-02 8.73226620391871488635615605744533e-02 3.50493272843274087519072068630521e-03
            4.21445202084233874750829329514090e-02 8.48561778910808195419690491689835e-01 3.58340542240752346625498603316373e-03
            8.48561797496106828830875201674644e-01 4.21445420914689297675614909621800e-02 3.58340680313374735851961183641379e-03
            8.47792133386372048775569965073373e-01 1.06743594247183659629030216819956e-01 3.84010356850248563143068736280838e-03
            1.06743588939808814464171859981434e-01 8.47792132814562227061117027915316e-01 3.84010458562234662308054744528363e-03
            1.83396652199136978023119581848732e-01 4.16340521608503724704242188181524e-02 4.61309564035336083093197245830197e-03
            4.16340541166712388432280533834273e-02 1.83396519692967219228307840239722e-01 4.61309657864100112339977854958306e-03
            7.61163225156013090177964386384701e-01 1.94159920285148612784098531847121e-01 4.89584958933319408513895609758038e-03
            1.94159925414428363721341952441435e-01 7.61163215393799386099260573246283e-01 4.89585049984232138081674534646481e-03
            7.57937874717249937539520487916889e-01 4.39826608586334974471121483929892e-02 4.94081878355304149330606833245838e-03
            4.39826512394945398631840305370133e-02 7.57937824230775603062681966548553e-01 4.94081916691774952754556693435006e-03
            3.69760535918213575912183443961112e-02 5.36318607643585854027890036377357e-01 4.97015977548178906203490967641301e-03
            5.36318713434215554869410880201031e-01 3.69760780935089808374449660277605e-02 4.97016214441264522760910438137216e-03
            1.00125694892088765164395169904310e-01 7.91226709354494128767498750676168e-01 5.17954596210734238836970177999319e-03
            7.91226669352416922542658994643716e-01 1.00125755467330726511399063838326e-01 5.17954837232312802208777924306560e-03
            3.79866714177100953109267322815867e-02 4.15741312855805678072584896654007e-01 5.22357678598892993110380800203529e-03
            4.15741402896513068121464584692148e-01 3.79867061534608085882069872241118e-02 5.22358129888737289514555328651113e-03
            6.50710649146250763585896947915899e-01 4.20141226713095228273431303023244e-02 5.37161434711807932174121305024528e-03
            4.20141133438424047019665863444970e-02 6.50710564508427236063425880274735e-01 5.37161465017301116564585328205794e-03
            4.25548444253750074439679451643315e-02 2.92062602348381217254313924058806e-01 5.55545332589627970792944822164827e-03
            2.92062710724053598010385712768766e-01 4.25548546752879397470081812571152e-02 5.55545400506844613136570032452255e-03
            5.38972953818032118888936565781478e-01 4.19303146900484813297538266851916e-01 5.58363263638368342856033166299312e-03
            4.19303182848857369791062410513405e-01 5.38972909361016605522820555052022e-01 5.58363446847851430804299255328260e-03
            6.54947200970027787469973645784194e-01 3.00735263616236814243620756315067e-01 5.61897312364901483500156942341164e-03
            3.00735279091710550058991202604375e-01 6.54947181273061840300897529232316e-01 5.61897451099071295471665266063610e-03
            3.75240077158565776827003901416901e-01 3.45398013075200016608334863121854e-01 5.74253489611704220907384410566010e-03
            3.45398028278600000007259041012730e-01 3.75240069567314926679557629540795e-01 5.74258486095953826766624672472972e-03
            9.94532168761113610777613303071121e-02 1.59830869518772511472803898868733e-01 5.81995940255617544289989950812014e-03
            1.59830935958481357150873236605548e-01 9.94531960132452180767259619642573e-02 5.81996068764666866363066688450090e-03
            1.79732666166714710742624561135017e-01 7.12458543092414076092211416835198e-01 6.73707999118512528002877814969906e-03
            7.12458446194299988007969659520313e-01 1.79732772223980236114826425364299e-01 6.73708267767895736116701144169383e-03
            1.06606567863572837540075965989672e-01 7.00170178417466426701309956115438e-01 7.01096895025934974121639697841601e-03
            7.00170190409589277003021834389074e-01 1.06606585567697198513315015588887e-01 7.01096911517204521985258125482687e-03
            9.93303629800770110369612098111247e-02 6.06564798479641065220846485317452e-01 7.18815675429640595084368470679692e-03
            6.06564805252115091249720535415690e-01 9.93303896768639588144012009252037e-02 7.18815968178396300786170414198750e-03
            1.02322354270440840640077340140124e-01 2.53338157952816500628756557489396e-01 7.47452072656893014290790944187393e-03
            2.53338232493812542944766619257280e-01 1.02322382618884696880989793044137e-01 7.47452306896452707551947725050923e-03
            6.16622671521702914176898957521189e-01 2.76950206057526682190683686712873e-01 7.72510896290271031894292264041724e-03
            2.76950069310862967419950564362807e-01 6.16622790062359649176926268410170e-01 7.72510964889170769626103307814446e-03
            9.04184571873021675214587844493508e-02 4.98152263700143593805336195146083e-01 7.85077542720023953404240302234030e-03
            4.98152276724787179684739157892182e-01 9.04185045149451249724137369412347e-02 7.85077684888205520885584576262772e-03
            9.28231860168124284227175735395576e-02 3.73841851690821957987509449594654e-01 7.97978882559495679427374881242940e-03
            3.73841869922919289859208902271348e-01 9.28232584789866893570930983514700e-02 7.97979170945421756311333183475654e-03
            2.52167884040700007730606557743158e-01 2.52168092569632074084751138798310e-01 8.03574810155277367840209734595192e-03
            5.08750021870755353425863631855464e-01 3.90558054432983914328048058450804e-01 8.25989003469931523659930405756313e-03
            3.90557911673060198509688234480564e-01 5.08750143766099505171496275579557e-01 8.25989079572204618062691849900148e-03
            1.70614146909608382918221991531027e-01 5.26673803955435726464884282904677e-01 8.90422738971193873658283024496995e-03
            5.26673776131196036409676253242651e-01 1.70614225753710269106733221633476e-01 8.90423190135420337953942038211608e-03
            3.48758152762900008880819768819492e-01 2.58805508488593516425169127614936e-01 9.14352974994153515353456640468721e-03
            2.58805359601724960239010897566914e-01 3.48758349170337211830883461516351e-01 9.14353788011750510944519731992841e-03
            1.69661455805267119556489774367947e-01 3.01352218396394311294983481275267e-01 9.14944117476145304945767122717371e-03
            3.01352180687456117080813555730856e-01 1.69661596321832502942683618130104e-01 9.14945134722811720395796442062419e-03
            2.58020240975833925478610808568192e-01 4.58474177447841080290658055673703e-01 9.24864200284901058490838465786510e-03
            4.58474086019770632471903581972583e-01 2.58020381901117201728368399926694e-01 9.24864437646179780860844488188377e-03
            1.84889868349759234344986680298462e-01 1.84889870455055865816618165808904e-01 9.35134059468711344165203769307482e-03
            6.13074033846504717892855751415482e-01 1.92161199406937599487577017498552e-01 9.38875645793730943811095102091713e-03
            1.92161175099370584717917154193856e-01 6.13074039838860329609815380536020e-01 9.38875781324003538808042890195793e-03
            4.18054116059869362853618213193840e-01 1.65061333641641855285442375134153e-01 9.72219233716051234317490070679924e-03
            1.65061264203588409227307920446037e-01 4.18054119924393174922983007490984e-01 9.72219270854397105141586621357419e-03
            5.15920573962530881750865319190780e-01 2.98271900522946442624316887304303e-01 9.81764108870523434513533089784687e-03
            2.98271893574985735853033474995755e-01 5.15920553436231665855871142412070e-01 9.81764506293797065139550284129655e-03
            4.09889460233993307980426834546961e-01 4.09889431779187407567377476880210e-01 9.96917199578724148867348020530699e-03
            ];
        
    case  25
        
        % ALG. DEG.:   25
        % PTS CARD.:  120
        % NEG. W.  :    0
        % OUT PTS. :    3
        % M.E.INF. : 7.0e-16
        
        xyw=[
            8.28815950327991060597732086989708e-03 9.84820276886916201597443887294503e-01 3.71835446477150599869226921256882e-04
            4.61842203024124764443314461459522e-01 5.38157796975900049041285910789156e-01 3.72225881549855474282606504488058e-04
            7.10664412391408435720752123643251e-03 8.08423613900631152229259157593333e-03 3.75148609496564391235784441391843e-04
            9.84761314169917412186805449891835e-01 7.00157551338421221126528948275336e-03 3.76480207818129187970906945892580e-04
            5.37444786904896942836273865395924e-01 4.62555213095117545574197492896928e-01 3.82971717875529491480202093356411e-04
            5.14426194399587329140975825306526e-15 4.88767688014002554819370516270283e-01 5.75815862178295691936602551663782e-04
            4.91413192936144105349427491091774e-01 -2.62191835077036522842371265891708e-14 5.91226676037731038379496872892105e-04
            7.03459370203730286613685862562306e-03 9.57415805369697281435037439223379e-01 7.18778590266164717215158219687510e-04
            9.56473471422832233201916096732020e-01 3.64655449485063384851279977283411e-02 7.46562218380497372854920357099218e-04
            3.70198792044907007414700217395875e-02 7.09085771655651366657258094505778e-03 7.59604068427608184431443216766411e-04
            1.02412454274703856871298057740205e-01 8.93612559493700997492737769789528e-01 8.02311492208139802502497861524944e-04
            5.92806581150900013810201016895007e-01 4.94517055997421069862829412500105e-03 9.25739960865503775169771394359941e-04
            5.09484223714734389687475157870722e-03 9.96676659188651253851176647913235e-02 9.35179650872904034107635506956058e-04
            8.15620236892535682704519217622874e-03 4.15561148783834197195652393475029e-02 9.61313580576800491908628920612045e-04
            4.24936107568289617164580818098329e-02 9.49486526035189015715332061517984e-01 9.66769466711528500241223937905488e-04
            9.49554350084383824714961974677863e-01 8.17945072920253993253858482148644e-03 9.79813879438329836391030625009080e-04
            8.93278747123910199512408780719852e-01 5.32243262623153211288151709368321e-03 9.89332067196990478513685118855392e-04
            6.93176129272652947338473339300435e-03 9.06540102043311923907253913057502e-01 1.10080629309176111080281756926524e-03
            9.03583903066471405551851603377145e-01 8.94771171076942650746133267603000e-02 1.14767770433724081678705797315843e-03
            9.05665738208722287394536465399142e-02 7.05253420051081634373213447020134e-03 1.17559173586508493129787478181925e-03
            8.39293327869924249418076556139567e-03 6.66317993111099959868681708030635e-01 1.25037109543959731872142526754033e-03
            6.26124568607099973860385944135487e-01 9.21975831529366594840890769546604e-03 1.30969575389001903140129279279336e-03
            6.28015929789052865239717959866539e-03 8.33520746052701988482169781491393e-01 1.36055260231148725941219268520399e-03
            8.27253925736676443136730085825548e-01 1.66513493933040795935696110063873e-01 1.42328122281234169255181853941394e-03
            6.20058753533867922785871584778761e-03 7.42469325522900014391325385076925e-01 1.47768557472601792801558673318141e-03
            1.67690031118506527318956500494096e-01 6.57177435282130240068454440915957e-03 1.49219919217398423391673922822065e-03
            7.19935306956684839541082965297392e-01 6.43545349617986699175542497641800e-03 1.68155475718053531997142613363394e-03
            2.74974009023742749224794579276931e-01 7.18529612071858725386164223891683e-01 1.70769621561272046984802930325031e-03
            7.92575820045364476396088093679282e-03 1.76641137471416409532309899077518e-01 1.73828147779962012206556032367644e-03
            6.99812207516695237302650767219347e-03 2.70476725400404682808641609881306e-01 1.81150676605872822458320747784910e-03
            8.12524877326335270844026581471553e-01 8.22995332101647721068093943586064e-03 1.81820474032382993143097671406849e-03
            7.35369699704123847699976934677579e-03 5.93416787545230528522210988739971e-01 1.82522327116445954274526197735895e-03
            7.28366593541109663689780973072629e-01 2.64881755375208860492364237870788e-01 1.84011666939011206411969112650695e-03
            1.80064230456454726914117259184422e-01 8.11584897668173987916873102221871e-01 1.85299808137789467216149930095526e-03
            2.65810246776189362805098426179029e-01 6.85535254287219369961903225885180e-03 1.87230535840114350500107409658312e-03
            7.08923645195805307572811670979718e-03 3.75763265974363824817316981352633e-01 1.96510168149251250938047697047750e-03
            3.77405430204347802636277720012004e-01 6.14857353375700022013461421011016e-01 1.96554316057912089807380162653772e-03
            3.69649608667726101152162243579369e-02 9.21079230289327588465653207094874e-01 2.01265902664448303668875972505248e-03
            9.20319410980492857987655952456407e-01 4.26025082113667408201251873833826e-02 2.03607151181833638531326613474448e-03
            4.25477806431233118722445851744851e-02 3.72689941794178275502602559754450e-02 2.03697010379726498041641491454357e-03
            6.19127839498267773521433809946757e-01 3.72405571380885980214969777080114e-01 2.31110366530089131678615821385847e-03
            3.76269720917775551960460234113270e-01 8.14364220110922777673412298327094e-03 2.36686587913678648728565967473969e-03
            9.56111149690206385542978750891052e-02 8.77109837260114377777142635750351e-01 2.42830525341710019848751045401514e-03
            3.02473410377312491859047582920539e-02 9.43858903393274828452774727338692e-02 2.49383952877324380562051864274054e-03
            8.73990569175390308309658848884283e-01 3.13198990882912128852488820029976e-02 2.58419509183475818239750232407914e-03
            8.60413373495807265456392087799031e-01 1.04901978204618509527001890546671e-01 2.80658192914217882729666087016085e-03
            3.47307852352399665241122761472070e-02 8.60985646288632278100294570322149e-01 2.85772796861267792262184528340185e-03
            1.04360660834317178768948508604808e-01 3.57152881004371841600075754286081e-02 2.88876418716774423928184134524599e-03
            7.79762282475343737608852734410902e-01 1.87231819926519243990625795959204e-01 3.38939642392396952783051311541840e-03
            1.85865164256267152709245493724666e-02 4.83439767879348392121130473242374e-01 3.39962407249161943992299406147595e-03
            3.24585286618077825671058178613748e-02 7.78347491604180818924874074582476e-01 3.44387034591525816040746832413788e-03
            8.37129390115686100592995444458211e-01 8.04060570156074766456910651868384e-02 3.44902539855731007922834407963819e-03
            8.36602075314564763708347072679317e-02 8.42141481705122707523969438625500e-01 3.46022099759418734141180884478217e-03
            7.84070242500509301253330818326504e-02 8.49927089144874164938769922628126e-02 3.51585049942114294743711333524061e-03
            4.92923864845802028877841394205461e-01 4.89285591471027347942879259790061e-01 3.52478627522342675276734169642623e-03
            1.87063758407251617965627588091593e-01 3.45210858281027530014561932603101e-02 3.55010279976978150931476996277070e-03
            4.89263696702512351777158983168192e-01 1.90774755076611041149092784507957e-02 3.61296061293265744970848452055634e-03
            4.01982618371916830546375365429412e-02 1.69114318710903849263260667612485e-01 3.75614949097134156671962124107722e-03
            7.89425927886497902363771572709084e-01 4.12206731483492141987490242627246e-02 3.82044510304562281999274375721143e-03
            1.68626045642937899549096414375526e-01 7.89486064058546044286401865974767e-01 3.88876810422339154807436223393324e-03
            3.75090191317404897386467155229184e-01 5.89531827201280833605778752826154e-01 4.11427215001154786971904186998472e-03
            3.56362876879900564164316278947808e-02 3.68125621769863742294148778455565e-01 4.13189398931612401205670792592173e-03
            5.88754816480384635113409785844851e-01 3.59968962540969730778961377382075e-02 4.17118886126239488826294987688925e-03
            3.73308082181670955468355543871439e-02 6.79070467353293794054991394659737e-01 4.18523282463122408419975428728321e-03
            2.82076999337395051270505064167082e-01 3.73639992361457645708711083898379e-02 4.21686658401333736012617237065569e-03
            6.81927760331965671092291358945658e-01 2.80333034572545214402339297521394e-01 4.22205575412311004401821890041901e-03
            3.74938324381651408301330263839191e-02 2.63401618001446569117263152293162e-01 4.30217781727027046706490764904629e-03
            6.98407920412669636078817347879522e-01 3.64154673321932242924248157578404e-02 4.36702670660430759819181645298158e-03
            2.65439089407868589187700081311050e-01 6.98071743619317341966734602465294e-01 4.41659749036037684033484396195490e-03
            1.42984844080011913591832239944779e-01 7.61225461845349782308289832144510e-01 4.57419053687040891992277025224212e-03
            7.62355400764634705268463221727870e-01 9.43741220275100267667056641585077e-02 4.58942131146659621637740045230203e-03
            9.34222022748698310845583137052017e-02 1.47979983683175969622070056175289e-01 4.65981423802769561398617881309292e-03
            5.75900447992329755919627132243477e-01 3.82132964169802080700577562311082e-01 4.74452651474672158771195284998612e-03
            3.82242733252505628716022556545795e-01 4.26716362301194362949274818674894e-02 4.79619806445563765656681098903391e-03
            4.11414081674571288638198041098804e-02 5.71808287443200891075889558123890e-01 4.85201105110986269375317192498187e-03
            8.02462538378740236622022052870307e-02 7.70220438204229251866195227194112e-01 4.86800180481780555424853318413625e-03
            7.62522981940991995131184921774548e-01 1.55942057736158296021855562685232e-01 5.02137701236820156031059880774592e-03
            1.52494144513048351941009173060593e-01 8.42965421322266761361419185050181e-02 5.04184773330151100173557310313299e-03
            6.22159195833349928927979988202424e-02 4.53818131887360642906514840433374e-01 5.54355406903198500573326512608219e-03
            1.10953903607625831151395345841593e-01 4.58601407117075587915877576961066e-01 5.74256101270903714095528158622983e-03
            4.57562721205716171901656252885005e-01 4.79531356021019539159766509328620e-01 5.83662793497304478035347941045075e-03
            4.32286513637400005016786508349469e-01 1.23059123747196408338311357510975e-01 5.87207838345829135490161476695903e-03
            5.86500285024055267868448027002160e-01 8.34119779793435350834229780048190e-02 6.01705247543985952130007888172258e-03
            8.69359250817794138033178796831635e-02 6.75567701335066272960716560191941e-01 6.02276982383004484689070778813402e-03
            9.29594906936006132314176397812844e-02 2.32650089272664062356454905966530e-01 6.14192623703284108077626868293919e-03
            6.66193214145382905932990524888737e-01 2.44829400740542629799634255505225e-01 6.16340789298672371565590566433457e-03
            4.78030636222676108548768070249935e-01 6.61749044834882976795498166211473e-02 6.16891325129698131490840751212090e-03
            4.37221529457722968281530029344140e-01 4.44214558524454405930015354897478e-01 6.24261506775146239600626785204440e-03
            6.77922450466891035603111959062517e-01 9.29096534576604599076077306563093e-02 6.25066360203841049625328452066242e-03
            2.42343125565994327308416700361704e-01 8.89793655129190036445763212213933e-02 6.26227173565197751253785796166085e-03
            2.28892542030518980622488811604853e-01 6.78005308167170972488690949830925e-01 6.27340625313686957309489145018233e-03
            3.31506504995856432493894772051135e-01 5.84738155974041440288146986858919e-01 6.28705190565876422675861689981502e-03
            3.42420052660672158406640619432437e-01 5.13924572273599999938653581921244e-01 6.37525726118703776890139778288358e-03
            8.62630046474948397605686523093027e-02 3.34097624923409219910297451860970e-01 6.41361279948181144117125285220027e-03
            5.11318894663500000952183199842693e-01 1.38015472055394355921364990535949e-01 6.44936876574064409800568498098983e-03
            1.53897784100073625745963568078878e-01 6.78806261956247491973215346661163e-01 6.75019384983002734834656166640343e-03
            6.77995134847147129519839836575557e-01 1.66335892526875278596421026122698e-01 6.86078842108803904631386672008375e-03
            1.66460046941077521864826849196106e-01 1.58221450484884951226405291890842e-01 6.92681003719222265074817457275458e-03
            9.50910318887985689384478860119998e-02 5.66659033254306332416661007300718e-01 6.95711038409810821059320318227037e-03
            3.43604813671150643994423035110231e-01 9.78960873456868746478320986170729e-02 7.18018452762302864661458912109993e-03
            5.56041702536571924575525827094680e-01 3.46891782094658818191845739420387e-01 7.22067087390573705135699000834393e-03
            1.45240402951303859691023490086081e-01 3.59953449105217593295691358434851e-01 7.33256824396508535834904307648685e-03
            1.61968515623819420135021118767327e-01 5.81013137333045603938330714299809e-01 7.97257198891866865342326065047018e-03
            5.80016484426167822086028991179774e-01 2.56067464067228411916943286996684e-01 7.97709150714511795732253318647054e-03
            2.45020122328836575942290210150531e-01 5.88146955210173794093009291827912e-01 8.02347400823490847221464150607062e-03
            2.55762189179423060636509035248309e-01 1.65224406504668047679373898972699e-01 8.04046521949311476795685393881286e-03
            2.20523998551101912957150830152386e-01 3.49650746610609075570863524262677e-01 8.06060318833910945113263579742124e-03
            4.94018311128501474982499530597124e-01 2.54944844845292173385331579993363e-01 8.17681116051974243852384915953735e-03
            2.53157068979825539134509426730801e-01 2.54336911501687523706749516350101e-01 8.24865791736491271723963336626184e-03
            5.84689111635679714673585749551421e-01 1.66660391647927841329845932705211e-01 8.29570240063638719707483915044577e-03
            1.66033360227843096446065374038881e-01 2.52324019170541480328040506719844e-01 8.37142906627554549137126826963140e-03
            2.50542629246117087848944038341870e-01 4.95900762752834678437352522450965e-01 8.38671181980161183078248399169752e-03
            3.51933680218193378763658074603882e-01 1.80538036780004129688848024670733e-01 8.42622607469532651502941433818705e-03
            3.50266883541880902352971816071658e-01 4.35858232988066807145344228047179e-01 8.50903656917300146855787090771628e-03
            4.40089248551157830036117957206443e-01 2.12057610494083004226695265970193e-01 8.56163088307875831672877353639706e-03
            4.68085547154612702236420318513410e-01 3.55268157077371138363730551645858e-01 8.63822043126822973480027911818979e-03
            1.77023776394722193172626134582970e-01 4.67035292226636644663528841192601e-01 8.91957189257099047896559795844951e-03
            3.90092077950076832060233300580876e-01 3.32315281929987604847553939180216e-01 9.11640562539701992950114117775229e-03
            2.80584777412015762276809027753188e-01 3.89804117668033467403176928200992e-01 9.12931771766211860863471372340427e-03
            3.36152334743991199061952102056239e-01 2.77850004435603759045392280313536e-01 9.29812027543928942829865036401316e-03
            ];
        
        
    otherwise
        xyw=[];
end




pointset_stats=[
    2    3 2.12e-16    0    0
    4    6 2.22e-16    0    0
    5   10 1.77e-13    0    0
    7   15 4.59e-13    0    0
    9   21 3.45e-16    0    0
    11   28 4.34e-16    0    0
    13   36 5.46e-16    0    0
    14   45 4.53e-13    0    0
    16   55 7.82e-13    0    0
    18   66 4.77e-16    0    0
    20   78 6.04e-16    0    0
    21   91 5.97e-16    0    0
    23  105 8.74e-16    0    0
    25  120 6.97e-16    0    3
    ];




