
function [xyw_bar,pointset_stats]=set_taylor_wingate_bos_barycentric(deg)

%--------------------------------------------------------------------
% CUBATURE RULE ON THE SIMPLEX [0 0; 0 1; 1 0] WITH ADE EQUAL TO deg.
% THE CUBATURE RULE IS DESCRIBED IN BARYCENTRICAL FORM.
%--------------------------------------------------------------------
% INPUT:
% deg: algebraic degree of precision (ADE) of the rule.
%--------------------------------------------------------------------
% OUTPUT:
% xyw_bar: MATRIX OF NODES xyw(:,1:3) IN BARYCENTRICAL COOR. AND WEIGHTS xyw(:,4).
%      REMARK: The sum of the weights makes 1.
%
% pointset_stats:
%    statistics of the pointset stored in a matrix. Each rows contains
%    degree, cardinality of the full pointset, moment error w.r.t.
%    orthn. Dubiner basis, number of points with negative weights,
%    number of points outside the domain.
%--------------------------------------------------------------------
% SOURCE:
%  Mark A. Taylor, Beth A. Wingate, Len P. Bos,
%  "Several new quadrature formulas for polynomial integration in the triangle",
%  unpublished note, see arXiv.
%--------------------------------------------------------------------
% | DEG | CARD | W- | PT.O|   M.E.   |Q.|
%--------------------------------------------------------------------
% |   2 |   3 |   0 |   0 | 1.73e-13 |PI|
% |   4 |   6 |   0 |   0 | 1.87e-13 |PI|
% |   5 |  10 |   0 |   0 | 1.77e-13 |PI|
% |   7 |  15 |   0 |   0 | 4.59e-13 |PI|
% |   9 |  21 |   0 |   0 | 3.33e-13 |PI|
% |  11 |  28 |   0 |   0 | 5.89e-13 |PI|
% |  13 |  36 |   0 |   0 | 3.94e-13 |PI|
% |  14 |  45 |   0 |   0 | 4.53e-13 |PI|
% |  16 |  55 |   0 |   0 | 7.82e-13 |PI|
% |  18 |  66 |   0 |   0 | 5.01e-13 |PI|
% |  20 |  78 |   0 |   0 | 7.68e-13 |PI|
% |  21 |  91 |   0 |   0 | 8.94e-13 |PI|
% |  23 | 105 |   0 |   0 | 7.02e-13 |PI|
% |  25 | 120 |   0 |   0 | 5.66e-13 |PI|
%--------------------------------------------------------------------
% LEGEND:
%--------------------------------------------------------------------
% DEG  : algebraic degree of precision
% CARD : cardinality of the pointset
% W-   : negative weights
% PT.O : points outside the simplex [0 0; 1 0; 0 1].
% M.E. : maximum moment error in inf norm.
% Q.: quality of the rule.
%        PI: pos. weights, points in simplex
%        NI: some neg. weights, points in simplex
%        P0: pos. weights, some points out of the simplex
%        N0: some neg. weights, some points out of the simplex
%--------------------------------------------------------------------


switch deg
    
    case   2
        
        % ALG. DEG.:    2
        % PTS CARD.:    3
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.7e-13
        
        xyw_bar=[
            1.66666666666699991861122498448822e-01 6.66666666666700047372273729706649e-01 1.66666666666599905255452540586703e-01 3.33333333333350023686136864853324e-01
            6.66666666666700047372273729706649e-01 1.66666666666699991861122498448822e-01 1.66666666666599960766603771844530e-01 3.33333333333350023686136864853324e-01
            1.66666666666699991861122498448822e-01 1.66666666666699991861122498448822e-01 6.66666666666599905255452540586703e-01 3.33333333333350023686136864853324e-01
            ];
        
    case   4
        
        % ALG. DEG.:    4
        % PTS CARD.:    6
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.9e-13
        
        xyw_bar=[
            9.15762135097999974142624068917939e-02 9.15762135097999974142624068917939e-02 8.16847572980400116193777648732066e-01 1.09951743655299999380403619397839e-01
            8.16847572980500036265993912820704e-01 9.15762135097999974142624068917939e-02 9.15762135096999663197436802875018e-02 1.09951743655299999380403619397839e-01
            9.15762135097999974142624068917939e-02 8.16847572980500036265993912820704e-01 9.15762135097000218308949115453288e-02 1.09951743655299999380403619397839e-01
            1.08103018168100001306086710428644e-01 4.45948490915999973260852584644454e-01 4.45948490915900053188636320555815e-01 2.23381589678000008758473882153339e-01
            4.45948490915999973260852584644454e-01 1.08103018168100001306086710428644e-01 4.45948490915900053188636320555815e-01 2.23381589678000008758473882153339e-01
            4.45948490915999973260852584644454e-01 4.45948490915999973260852584644454e-01 1.08103018168000053478294830711093e-01 2.23381589678000008758473882153339e-01
            ];
        
    case   5
        
        % ALG. DEG.:    5
        % PTS CARD.:   10
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.8e-13
        
        xyw_bar=[
            0.00000000000000000000000000000000e+00 1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 1.31356049751999992558637586625991e-02
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 1.31358306033999994860694826570580e-02
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 1.00000000000000000000000000000000e+00 1.37081973800000003066434928200579e-02
            2.67327353118499977391309130325681e-01 6.72819921871000037327803511288948e-01 5.98527250104999852808873583853710e-02 1.17419193291149998037425916663778e-01
            6.72817552946100017052799557859544e-01 2.67328859948199992757622567296494e-01 5.98535871056999901895778748439625e-02 1.17420611913400002590535109447956e-01
            6.49236350053999933251702714187559e-02 6.71653011149400014367927269631764e-01 2.63423353845199992306902458949480e-01 1.24012589655700000701443741490948e-01
            6.71649853904199956033949092670809e-01 6.49251690029000044779294853469764e-02 2.63424977092900025610333614167757e-01 1.24015246126049999420004610328760e-01
            6.54032456799999994956351656583138e-02 2.69378936645299993379865099996096e-01 6.65217817674700007124499734345591e-01 1.25930230276450005888477789994795e-01
            2.69376706914000008108445172183565e-01 6.54054874919000051036732656939421e-02 6.65217805594099931276730330864666e-01 1.25933026682899990023045688758430e-01
            3.38673850389600017418700872440240e-01 3.38679989302699990805223251300049e-01 3.22646160307699936264924645001884e-01 2.25289469095699995682124949780700e-01
            ];
        
    case   7
        
        % ALG. DEG.:    7
        % PTS CARD.:   15
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.6e-13
        
        xyw_bar=[
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 5.12790870460000033914660377831751e-03
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 1.00000000000000000000000000000000e+00 5.12790870460000033914660377831751e-03
            0.00000000000000000000000000000000e+00 1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 5.12790870460000033914660377831751e-03
            7.83965665101200026576577783998800e-01 4.21382841641999988935296528325125e-02 1.73896050734599960652104755354230e-01 5.58023523323500011961151301420614e-02
            1.73896050734500012824312875636679e-01 7.83965665101200026576577783998800e-01 4.21382841642999883546849559934344e-02 5.58023523323500011961151301420614e-02
            1.73896050734500012824312875636679e-01 4.21382841641999988935296528325125e-02 7.83965665101300057671096510603093e-01 5.58023523323500011961151301420614e-02
            4.21382841641999988935296528325125e-02 1.73896050734500012824312875636679e-01 7.83965665101300057671096510603093e-01 5.58023523323500011961151301420614e-02
            7.83965665101200026576577783998800e-01 1.73896050734500012824312875636679e-01 4.21382841642999605991093403645209e-02 5.58023523323500011961151301420614e-02
            4.21382841641999988935296528325125e-02 7.83965665101200026576577783998800e-01 1.73896050734600016163255986612057e-01 5.58023523323500011961151301420614e-02
            4.74388086175200007765795362502104e-01 4.74388086175200007765795362502104e-01 5.12238276495999844684092749957927e-02 8.39887797667499963472081958570925e-02
            4.74388086175200007765795362502104e-01 5.12238276497000016851401937856281e-02 4.74388086175099976671276635897812e-01 8.39887797667499963472081958570925e-02
            5.12238276497000016851401937856281e-02 4.74388086175200007765795362502104e-01 4.74388086175099976671276635897812e-01 8.39887797667499963472081958570925e-02
            2.38561530018099987593416244635591e-01 5.22876939963899944885383774817456e-01 2.38561530018000067521199980546953e-01 1.32611940197299993382529237351264e-01
            5.22876939963899944885383774817456e-01 2.38561530018099987593416244635591e-01 2.38561530018000067521199980546953e-01 1.32611940197299993382529237351264e-01
            2.38561530018099987593416244635591e-01 2.38561530018099987593416244635591e-01 5.22876939963800024813167510728817e-01 1.32611940197299993382529237351264e-01
            ];
        
    case   9
        
        % ALG. DEG.:    9
        % PTS CARD.:   21
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.3e-13
        
        xyw_bar=[
            4.51890097843999985482987824525480e-02 4.51890097843999985482987824525480e-02 9.09621980431200016781190242909361e-01 2.59935710322999995181003640709605e-02
            4.51890097843999985482987824525480e-02 9.09621980431200016781190242909361e-01 4.51890097843999916094048785453197e-02 2.59935710322999995181003640709605e-02
            9.09621980431200016781190242909361e-01 4.51890097843999985482987824525480e-02 4.51890097843999846705109746380913e-02 2.59935710322999995181003640709605e-02
            7.47512472733899979715488370857202e-01 3.04243617287999999765357728165327e-02 2.22063165537300016838528904372652e-01 3.53517050891999987105585034896649e-02
            2.22063165537299989082953288743738e-01 3.04243617287999999765357728165327e-02 7.47512472733899979715488370857202e-01 3.53517050891999987105585034896649e-02
            7.47512472733899979715488370857202e-01 2.22063165537299989082953288743738e-01 3.04243617288000312015583403990604e-02 3.53517050891999987105585034896649e-02
            2.22063165537299989082953288743738e-01 7.47512472733899979715488370857202e-01 3.04243617288000312015583403990604e-02 3.53517050891999987105585034896649e-02
            3.04243617287999999765357728165327e-02 7.47512472733899979715488370857202e-01 2.22063165537299989082953288743738e-01 3.53517050891999987105585034896649e-02
            3.04243617287999999765357728165327e-02 2.22063165537299989082953288743738e-01 7.47512472733899979715488370857202e-01 3.53517050891999987105585034896649e-02
            1.36991201264899997713797574760974e-01 2.18290070971399990940042812326283e-01 6.44718727763700094612886459799483e-01 4.54695380475999971769951457645220e-02
            6.44718727763699983590583997283829e-01 2.18290070971399990940042812326283e-01 1.36991201264900025469373190389888e-01 4.54695380475999971769951457645220e-02
            1.36991201264899997713797574760974e-01 6.44718727763699983590583997283829e-01 2.18290070971400074206769659213023e-01 4.54695380475999971769951457645220e-02
            2.18290070971399990940042812326283e-01 6.44718727763699983590583997283829e-01 1.36991201264900053224948806018801e-01 4.54695380475999971769951457645220e-02
            2.18290070971399990940042812326283e-01 1.36991201264899997713797574760974e-01 6.44718727763700094612886459799483e-01 4.54695380475999971769951457645220e-02
            6.44718727763699983590583997283829e-01 1.36991201264899997713797574760974e-01 2.18290070971400018695618427955196e-01 4.54695380475999971769951457645220e-02
            3.69603304334000029585105551177548e-02 4.81519834783300015867979482209194e-01 4.81519834783300015867979482209194e-01 5.16172025690000008113500484796532e-02
            4.81519834783300015867979482209194e-01 3.69603304334000029585105551177548e-02 4.81519834783299904845677019693539e-01 5.16172025690000008113500484796532e-02
            4.81519834783300015867979482209194e-01 4.81519834783300015867979482209194e-01 3.69603304333999127528898043237859e-02 5.16172025690000008113500484796532e-02
            4.03603979817900027082799851996242e-01 1.92792040364099998006608416289964e-01 4.03603979817999891643864884827053e-01 9.40800734583499970176134752364305e-02
            4.03603979817900027082799851996242e-01 4.03603979817900027082799851996242e-01 1.92792040364199890323249064749689e-01 9.40800734583499970176134752364305e-02
            1.92792040364099998006608416289964e-01 4.03603979817900027082799851996242e-01 4.03603979817999947155016116084880e-01 9.40800734583499970176134752364305e-02
            ];
        
    case  11
        
        % ALG. DEG.:   11
        % PTS CARD.:   28
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.9e-13
        
        xyw_bar=[
            0.00000000000000000000000000000000e+00 9.45170445017399996068263590132119e-01 5.48295549826000039317364098678809e-02 5.70412470164999990673848628830456e-03
            9.45170445017299964973744863527827e-01 0.00000000000000000000000000000000e+00 5.48295549827000350262551364721730e-02 5.70412470164999990673848628830456e-03
            9.28900240571900015495998559345026e-01 6.85505797224000013034839184911107e-02 2.54917970569998320051752216386376e-03 6.63456428599999963741096209446368e-03
            6.85505797224000013034839184911107e-02 9.28900240571699953306961106136441e-01 2.54917970590007314513059100136161e-03 6.63456428599999963741096209446368e-03
            2.43268355615000007874382248473921e-02 2.43268355616000006569343838691566e-02 9.51346328876900071414013382309349e-01 7.79328866750000036345280562954940e-03
            1.27966283533500008173788842213980e-01 2.77838749488000004428389644317576e-02 8.44249841517699994852819145307876e-01 2.04137390214000000177030358372576e-02
            2.77838749488000004428389644317576e-02 1.27966283533699987096099448535824e-01 8.44249841517500043686084154614946e-01 2.04137390214500016871745913249470e-02
            2.87083428359999996704043923045901e-02 7.49834758865699946284166799159721e-01 2.21456898298300064453769664396532e-01 2.89924832557999986870633790658758e-02
            7.49834758865600026211950535071082e-01 2.87083428359999996704043923045901e-02 2.21456898298399984525985928485170e-01 2.89924832557999986870633790658758e-02
            7.22800790970700046678132366650971e-01 2.49760206238500009900960208142351e-01 2.74390027907999434209074252066785e-02 3.00692623831500006892358811683152e-02
            2.49760206238600013239903319117730e-01 7.22800790970700046678132366650971e-01 2.74390027906999955931155454891268e-02 3.00692623831500006892358811683152e-02
            8.65562992839000039380081830131530e-02 8.32551385699699997289258135424461e-01 8.08923150164000404060971050057560e-02 3.12636944216500012316828360781074e-02
            8.32551385699800028383776862028753e-01 8.65562992839000039380081830131530e-02 8.08923150162999676782149549580936e-02 3.12636944216500012316828360781074e-02
            3.06161915767199976912138481566217e-01 3.03526617491000003101753179635125e-02 6.63485422483700015838792296563042e-01 3.19842160752000018164586947477801e-02
            3.03526617491000003101753179635125e-02 3.06161915767500014684543430121266e-01 6.63485422483399922555236116750166e-01 3.19842160752000018164586947477801e-02
            4.86861059504699988931264442726388e-01 4.86861059504699988931264442726388e-01 2.62778809905999666263198832893977e-02 3.30662936080500027835604726078600e-02
            6.65790429301699959729887723369757e-01 1.76545615421900004005095752290799e-01 1.57663955276400036265016524339444e-01 3.34251618410000006509363856821437e-02
            1.76545615422100010682981974241557e-01 6.65790429301600039657671459281119e-01 1.57663955276299949659346566477325e-01 3.34251618410500023204079411698331e-02
            2.93121007360000011832124044985903e-02 5.29565748866899999036661483842181e-01 4.41122150397100010188466967520071e-01 3.43452152988499986863146773430344e-02
            5.29565748866700047869926493149251e-01 2.93121007360000011832124044985903e-02 4.41122150397299961355201958213001e-01 3.43452152988499986863146773430344e-02
            1.44467382439100011648491772575653e-01 1.44467382439100011648491772575653e-01 7.11065235121799865680713992333040e-01 5.01358771929499985597900035827479e-02
            3.29974011141099976551771533195279e-01 5.36181572905000014550580544892000e-01 1.33844415953899953386496690654894e-01 5.71568392049500023643915369575552e-02
            5.36181572905199965717315535584930e-01 3.29974011140900025385036542502348e-01 1.33844415953900008897647921912721e-01 5.71568392049500023643915369575552e-02
            5.51150751686199980028391109954100e-01 1.43779086192300009638955771151814e-01 3.05070162121500010332653118894086e-01 6.11824073375999977209005464828806e-02
            1.43779086192300009638955771151814e-01 5.51150751686199980028391109954100e-01 3.05070162121499954821501887636259e-01 6.11824073375999977209005464828806e-02
            3.34806658732699990999037709116237e-01 1.52961943716100001733337876430596e-01 5.12231397551200062778775645710994e-01 6.97211167088999939078775014422718e-02
            1.52961943716100001733337876430596e-01 3.34806658732699990999037709116237e-01 5.12231397551199951756473183195340e-01 6.97211167088999939078775014422718e-02
            3.43018349814699974320575392994215e-01 3.43018349814699974320575392994215e-01 3.13963300370600106870000445269397e-01 8.72188914591000008513788088748697e-02
            ];
        
    case  13
        
        % ALG. DEG.:   13
        % PTS CARD.:   36
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.9e-13
        
        xyw_bar=[
            2.42935351589999987131918146587850e-02 9.49305929384600011289307985862251e-01 2.64005354563999761197123916645069e-02 8.31204993784999922323830645609632e-03
            2.65193427721999996915513975181966e-02 2.42695130639999992472333190107747e-02 9.49211144163799924733382340491517e-01 8.34058498889999992698118802536555e-03
            9.49212602355099965834028807876166e-01 2.65067966436999986890032943165352e-02 2.42806010012000354769678978072989e-02 8.34152845335000073245801388566179e-03
            3.37757637490000012236990301062178e-03 4.76731641236300018871219208449475e-01 5.19890782388799932434153561189305e-01 8.78404350414999958440720462249374e-03
            4.75767229810100000797490338300122e-01 5.19892182910200006951129125809530e-01 4.34058727969999225138053589034826e-03 9.22373309225000062583355031620158e-03
            5.19078319347100025638042097853031e-01 5.59127062019999961084426587376583e-03 4.75330410032699990363624920064467e-01 9.89712050940000018228737133085815e-03
            8.61683974532099994370071271987399e-01 1.33996048618000001412253752164361e-02 1.24916420606100003753979876819358e-01 1.01770197927500000156797455019841e-02
            1.24920975992600005732846568662353e-01 8.61305432133400050709326478681760e-01 1.37735918739999574356147604703438e-02 1.03426431969999999688214487036930e-02
            1.38565453861000005553183456186161e-02 1.24773371735799998294424995037843e-01 8.61370082878099907475188956595957e-01 1.04135683042999999114996967364277e-02
            2.11887064221999987800870712817414e-02 8.43843835122299990025851457176032e-01 1.34967458455499955682910240284400e-01 1.58909889139500007382288515600521e-02
            8.43229678721900022253521456150338e-01 1.35456364583000005152868538971234e-01 2.13139566950999725936100048784283e-02 1.60236017620500013902695712886270e-02
            1.35423179786500008647820436635811e-01 2.13482820656000008008312818219565e-02 8.43228538147899997490242185449461e-01 1.60303840572999997671210792304919e-02
            3.08885351067899993626753030184773e-01 2.21919663013999991785230037066867e-02 6.68922682630700093930897764948895e-01 2.15382979591500003935244933472859e-02
            6.68505759516900055494659227406373e-01 3.08901287938899982155049883658648e-01 2.25929525441999623502908889349783e-02 2.19236707669499993056394515633656e-02
            2.26545012556999998676321439461390e-02 6.69170994332100033119559157057665e-01 3.08174504412199956604467843135353e-01 2.19604836366499994726453337534622e-02
            2.80851540877199989942880620219512e-01 6.92471815510599997267604521766771e-01 2.66766436122000127895148580137175e-02 2.39975961845499997926456359209624e-02
            6.92244674905099999229207696771482e-01 2.68723345026000010049571642412047e-02 2.80882990592299985888047331172857e-01 2.41903130366500004888585806384071e-02
            2.68617447119000006539035751984557e-02 2.81009397322199983015877933212323e-01 6.92128857965900068371922770893434e-01 2.42433711687499997555761410694686e-02
            1.14177848546999999301831962839060e-01 7.97358141358599947778884597937576e-01 8.84640100944000806748590548522770e-02 2.78482244012000015875862857228640e-02
            7.97480792206100019825498748105019e-01 8.79806508790999947100530675925256e-02 1.14538556914799985464448184302455e-01 2.80513182177999989552752424515347e-02
            8.92807293894000014722323044225050e-02 1.14502056112799996401285795855074e-01 7.96217214497800029882057515351335e-01 2.82595061846500003599480521643272e-02
            1.05248789245500004585487374697550e-01 6.68690411992200028912236575706629e-01 2.26060798762300008135639473039191e-01 3.44644945334999994179803195493150e-02
            6.66302228073999969204521676147124e-01 2.27505163183199993204297584270535e-01 1.06192608742800037591180739582342e-01 3.58606668044499990455342697259766e-02
            2.30780373754700007804530059729586e-01 1.05457256122099995998553367826389e-01 6.63762370123199940685765341186197e-01 3.63726960488000017335252778138965e-02
            1.70505915754000009210145094584732e-01 5.17406439865800038546694850083441e-01 3.12087644380199979998735670960741e-01 3.94403668368499965568574339158658e-02
            5.08659397304299987219167178409407e-01 3.17052385520899981941056466894224e-01 1.74288217174800030839776354696369e-01 4.05057172755999977953322854773432e-02
            3.14182386228100019032183354283916e-01 1.81070636165899995217998252883262e-01 5.04746977605999957994242777203908e-01 4.12862649527500000234958577038924e-02
            4.61746081786399975577239729318535e-01 4.67859453980400019634799946288695e-01 7.03944642332000047879603243927704e-02 4.21022283665000024432956138298323e-02
            6.93087496081000037362329635470815e-02 4.62285604208500011669258356050705e-01 4.68405646183400026227872103845584e-01 4.21792766652500025759131574432104e-02
            4.65195525926800024230090002674842e-01 7.24357805669000043291561041769455e-02 4.62368693506299943685178277519299e-01 4.25984934244000024161636019925936e-02
            2.57862585789299991745338047621772e-01 6.13139503917700001700552547845291e-01 1.28997910293000006554109404532937e-01 4.51422664026000022841067504941748e-02
            6.11262776677899966060181213833857e-01 1.30036083460900009356464579468593e-01 2.58701139861200024583354206697550e-01 4.57141571742500016362065196062758e-02
            1.30518213593399989846943753946107e-01 2.58171382888400025024111528182402e-01 6.11310403518199985128944717871491e-01 4.58139532704500002102676603499276e-02
            4.28143799182799988933822987746680e-01 2.36200596981700011722793419721711e-01 3.35655603835499971587807976902695e-01 5.12786687447999989242219953666790e-02
            3.35699578373000018238769825984491e-01 4.31102630858800017765730672181235e-01 2.33197790768199963995499501834274e-01 5.16579830706499973369538736278628e-02
            2.30542429883599991891074409977591e-01 3.45601394937600003842703699774574e-01 4.23856175178800032021797505876748e-01 5.17927183596499987605632497889019e-02
            ];
        
    case  14
        
        % ALG. DEG.:   14
        % PTS CARD.:   45
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.5e-13
        
        xyw_bar=[
            0.00000000000000000000000000000000e+00 1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 5.30835599499999998863997152653837e-04
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 5.30835599499999998863997152653837e-04
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 1.00000000000000000000000000000000e+00 5.30835599499999998863997152653837e-04
            5.73330873025999984693434896598774e-02 1.51382269814000006147303167836071e-02 9.27528685716000000915926193556515e-01 6.57301180504999988524206244733250e-03
            5.73330873025999984693434896598774e-02 9.27528685716000000915926193556515e-01 1.51382269813999936758364128763787e-02 6.57301180504999988524206244733250e-03
            9.27528685716000000915926193556515e-01 5.73330873025999984693434896598774e-02 1.51382269814000006147303167836071e-02 6.57301180504999988524206244733250e-03
            1.51382269814000006147303167836071e-02 5.73330873025999984693434896598774e-02 9.27528685716000000915926193556515e-01 6.57301180504999988524206244733250e-03
            9.27528685716000000915926193556515e-01 1.51382269814000006147303167836071e-02 5.73330873025999984693434896598774e-02 6.57301180504999988524206244733250e-03
            1.51382269814000006147303167836071e-02 9.27528685716000000915926193556515e-01 5.73330873026000054082373935671058e-02 6.57301180504999988524206244733250e-03
            8.15962504071100047653430920036044e-01 1.65971996956500000841572273202473e-01 1.80654989723999515049968067614827e-02 1.21440963474499993846045242662512e-02
            8.15962504071100047653430920036044e-01 1.80654989724000000772541341120814e-02 1.65971996956499945330421041944646e-01 1.21440963474499993846045242662512e-02
            1.65971996956500000841572273202473e-01 8.15962504071100047653430920036044e-01 1.80654989723999515049968067614827e-02 1.21440963474499993846045242662512e-02
            1.80654989724000000772541341120814e-02 8.15962504071100047653430920036044e-01 1.65971996956500000841572273202473e-01 1.21440963474499993846045242662512e-02
            1.65971996956500000841572273202473e-01 1.80654989724000000772541341120814e-02 8.15962504071100047653430920036044e-01 1.21440963474499993846045242662512e-02
            1.80654989724000000772541341120814e-02 1.65971996956500000841572273202473e-01 8.15962504071100047653430920036044e-01 1.21440963474499993846045242662512e-02
            3.16547555637800015482241633435478e-01 1.86886898773000013906919747341817e-02 6.64763754484899993535407247691182e-01 1.58399933165999998174733320865926e-02
            6.64763754484899993535407247691182e-01 1.86886898773000013906919747341817e-02 3.16547555637800015482241633435478e-01 1.58399933165999998174733320865926e-02
            1.86886898773000013906919747341817e-02 6.64763754484899993535407247691182e-01 3.16547555637800015482241633435478e-01 1.58399933165999998174733320865926e-02
            1.86886898773000013906919747341817e-02 3.16547555637800015482241633435478e-01 6.64763754484899993535407247691182e-01 1.58399933165999998174733320865926e-02
            3.16547555637800015482241633435478e-01 6.64763754484899993535407247691182e-01 1.86886898772999909823511188733391e-02 1.58399933165999998174733320865926e-02
            6.64763754484899993535407247691182e-01 3.16547555637800015482241633435478e-01 1.86886898772999909823511188733391e-02 1.58399933165999998174733320865926e-02
            1.92662192491999985366479819504093e-02 4.90366890375399988588611677187146e-01 4.90366890375399988588611677187146e-01 1.74658973517999999147676248867356e-02
            4.90366890375399988588611677187146e-01 1.92662192491999985366479819504093e-02 4.90366890375400044099762908444973e-01 1.74658973517999999147676248867356e-02
            4.90366890375399988588611677187146e-01 4.90366890375399988588611677187146e-01 1.92662192492000783339278768835356e-02 1.74658973517999999147676248867356e-02
            8.75134669581000002036574869634933e-02 8.24973066083700024009317530726548e-01 8.75134669581999480314493666810449e-02 1.91832266972499984603039990815887e-02
            8.75134669581000002036574869634933e-02 8.75134669581000002036574869634933e-02 8.24973066083799944081533794815186e-01 1.91832266972499984603039990815887e-02
            8.24973066083700024009317530726548e-01 8.75134669581000002036574869634933e-02 8.75134669581999757870249823099584e-02 1.91832266972499984603039990815887e-02
            9.35526036219000017846170180746412e-02 2.07986542316699996035822550766170e-01 6.98460854061400016057348238973645e-01 2.89184745605000001478224191941990e-02
            9.35526036219000017846170180746412e-02 6.98460854061299984962829512369353e-01 2.07986542316800027130341277370462e-01 2.89184745605000001478224191941990e-02
            2.07986542316699996035822550766170e-01 9.35526036219000017846170180746412e-02 6.98460854061400016057348238973645e-01 2.89184745605000001478224191941990e-02
            6.98460854061299984962829512369353e-01 9.35526036219000017846170180746412e-02 2.07986542316800027130341277370462e-01 2.89184745605000001478224191941990e-02
            6.98460854061299984962829512369353e-01 2.07986542316699996035822550766170e-01 9.35526036220000190013479368644767e-02 2.89184745605000001478224191941990e-02
            2.07986542316699996035822550766170e-01 6.98460854061299984962829512369353e-01 9.35526036220000190013479368644767e-02 2.89184745605000001478224191941990e-02
            9.74892983467000040498362523067044e-02 5.38008859514899961951073237287346e-01 3.64501842138400089510241741663776e-01 3.62910843697000010110720324973954e-02
            3.64501842138300002904571783801657e-01 9.74892983467000040498362523067044e-02 5.38008859515000104067894426407292e-01 3.62910843697000010110720324973954e-02
            5.38008859514899961951073237287346e-01 9.74892983467000040498362523067044e-02 3.64501842138400033999090510405949e-01 3.62910843697000010110720324973954e-02
            5.38008859514899961951073237287346e-01 3.64501842138300002904571783801657e-01 9.74892983468000351443549789109966e-02 3.62910843697000010110720324973954e-02
            3.64501842138300002904571783801657e-01 5.38008859514899961951073237287346e-01 9.74892983468000906555062101688236e-02 3.62910843697000010110720324973954e-02
            9.74892983467000040498362523067044e-02 3.64501842138300002904571783801657e-01 5.38008859515000104067894426407292e-01 3.62910843697000010110720324973954e-02
            2.21714589487299995607116898099775e-01 5.56570821025299977691247477196157e-01 2.21714589487399971190484393446241e-01 4.48928262053499976813597527325328e-02
            5.56570821025299977691247477196157e-01 2.21714589487299995607116898099775e-01 2.21714589487400026701635624704068e-01 4.48928262053499976813597527325328e-02
            2.21714589487299995607116898099775e-01 2.21714589487299995607116898099775e-01 5.56570821025399897763463741284795e-01 4.48928262053499976813597527325328e-02
            3.86047166929600005236267179498100e-01 2.27905666140799989527465641003801e-01 3.86047166929600060747418410755927e-01 5.17272266808500025270234345953213e-02
            2.27905666140799989527465641003801e-01 3.86047166929600005236267179498100e-01 3.86047166929600005236267179498100e-01 5.17272266808500025270234345953213e-02
            3.86047166929600005236267179498100e-01 3.86047166929600005236267179498100e-01 2.27905666140800045038616872261628e-01 5.17272266808500025270234345953213e-02
            ];
        
    case  16
        
        % ALG. DEG.:   16
        % PTS CARD.:   55
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 7.8e-13
        
        xyw_bar=[
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 3.10129992550000005771537736620758e-04
            0.00000000000000000000000000000000e+00 1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 3.15758735599999988078595558604889e-04
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 1.00000000000000000000000000000000e+00 3.54330077950000001639485613935676e-04
            9.39886358357699958077091650920920e-01 4.98487446340000004557646562375339e-03 5.51287671789000427446936214437301e-02 2.75818580839999995604761018341833e-03
            5.43806683058000034369783293186629e-02 9.38640561861699973533745833265129e-01 6.97876983250000915148802960175090e-03 3.13462038279999988601987048753017e-03
            9.39400491640000022408596436207517e-03 5.26424462697000009048231561337161e-02 9.37963548813899983258579595712945e-01 3.92657044129999990150770017294235e-03
            1.64345086361999993818105991749690e-02 9.46903551735099990871447062090738e-01 3.66619396287000132161892906879075e-02 4.72757419320000028123418545078493e-03
            9.46948726986200028932216810062528e-01 3.63373677167000003196406510141969e-02 1.67139052970999707481425389232754e-02 4.89122556354999998146082518246658e-03
            4.26604005767999977516424792156613e-02 1.51224541798999998626351626285214e-02 9.42217145243300024937127545854310e-01 4.99308217444999983408715493737873e-03
            1.22269495439000005360119516240047e-02 8.69377351066400017032265168381855e-01 1.18395699389699959880317692295648e-01 6.87769094080000007590536270640769e-03
            8.67369652104699961903122584772063e-01 1.20491728577399995203656146713911e-01 1.21386193179000428932212685140257e-02 7.04895890199999999858881949421630e-03
            8.45674402138899994341159072064329e-01 1.57763967870000014481757233397730e-02 1.38549201074100014619006060456741e-01 7.48234321684999956897366146790773e-03
            1.39575963210299996974939062965859e-01 8.44812087037499992447919794358313e-01 1.56119497522000383327167583047412e-02 7.80487518060000041303370110767901e-03
            1.31782174323099993129559948101814e-01 1.35009605583999992006738466443494e-02 8.54716865118499979914190589624923e-01 7.88418466740000052628278837119069e-03
            1.57955126300000012196100840355939e-02 1.45527493853600004003467915936199e-01 8.38676993516400015593603711749893e-01 8.78972731915000048130703902415917e-03
            7.36546288443600039030911830195691e-01 1.55697540907999993492483170598462e-02 2.47883957465599963354563328721269e-01 1.02056920135000003119207789836764e-02
            1.39688430330000001866608272393933e-02 7.37983689445000035078692235401832e-01 2.48047467521999909223495706100948e-01 1.04781439308000001026099923251422e-02
            2.54789518603899978188565000891685e-01 7.29761568977099983612788491882384e-01 1.54489124190000381986465072259307e-02 1.05356706499000005866628626449710e-02
            7.31638652255499954968342990468955e-01 2.54307668331499980851617692678701e-01 1.40536794130000641800393168523442e-02 1.08823380101000007774691979989257e-02
            1.57253728950999989433245929149052e-02 2.69623979579100003789449147006962e-01 7.14650647525799986858885404217290e-01 1.11144204349499994832761728957848e-02
            2.66230284364699976151769078569487e-01 1.44783956308000005819280886498746e-02 7.19291320004500023266302832780639e-01 1.12093346841000002234167709502799e-02
            8.67350406521400052461956420302158e-01 5.91679410399999980429974755224976e-02 7.34816524385999425561522002681158e-02 1.15061308496499994924455734235380e-02
            7.41493666957000058248539176020131e-02 8.63478257506099966533952283498365e-01 6.23723757981999860078303754562512e-02 1.18406951249999999736806088890262e-02
            1.59285948359999991741364766539846e-02 4.19123895523800027440586291049840e-01 5.64947509640200018488087607693160e-01 1.28732321684000006190018439156120e-02
            1.56061028068000002405613102496318e-02 5.80922292114600002932434108515736e-01 4.03471605078600026317303672840353e-01 1.28978400803999998514015601358551e-02
            5.91009481748399956302364444127306e-01 1.59251452650999991700331293031923e-02 3.93065372986500016772026810940588e-01 1.29036163804999992610333947595791e-02
            4.03477149688900027513227541930974e-01 5.80670036810399969873230929806596e-01 1.58528135007000026135415282624308e-02 1.30171616029500000072394172434542e-02
            5.69474562852600008078240989561891e-01 4.14949514630200022580197583010886e-01 1.55759225171999693415614274272230e-02 1.32884070804499994367109394488580e-02
            6.78493700649999975649251382492366e-02 7.61218678590999953126328136931988e-02 8.56028762075899951611290816799738e-01 1.32892380915500003119023020303757e-02
            4.26596859027200014935488070477732e-01 1.57509692312000001979122032480518e-02 5.57652171741599977927705822366988e-01 1.33766164619000000463033828168591e-02
            6.70982507890000023920151761558373e-02 7.74189831242099946173595981235849e-01 1.58711917968900051434388842608314e-01 1.87893903320500016418392874584242e-02
            7.52831023147999989220124916755594e-01 8.19119495638999939313151799069601e-02 1.65257027288100016848559903337446e-01 1.91532947097499996191860560657005e-02
            7.75372778355699954211388558178442e-01 1.57712845729199996336333811086661e-01 6.69143759151000494522776307348977e-02 1.92424847512500003798852077352421e-02
            1.68907315778699995689748902805150e-01 7.50394309974199980040054924756987e-01 8.06983742471000242701961724378634e-02 1.94809912925999993704717638820512e-02
            1.68733583291899996714491294369509e-01 7.08311507268000045689504418078286e-02 7.60435265981299957083194840379292e-01 1.97302055773499983304386518057072e-02
            8.21244708436000003803911795330350e-02 1.76299662677100010821362729984685e-01 7.41575866479299961042670474853367e-01 2.06182389048999992198218933481257e-02
            6.28870536334499963260213917237706e-01 8.07744953317000025094785087276250e-02 2.90354968333800034230307574034669e-01 2.56436219241500014398571494211865e-02
            8.11413015265999987279954552832351e-02 3.05437358977599993092866270671948e-01 6.13421339495799911034623619343620e-01 2.58202820967500003279671005884666e-02
            2.96911206508000014014925227456843e-01 6.22748598887099991117111130733974e-01 8.03401946049000503791148730670102e-02 2.59115021134499988464394704124061e-02
            7.67542314170999995237565372008248e-02 6.24724714954600024441333516733721e-01 2.98521053628299948279334330436541e-01 2.64263994090499984568332081380504e-02
            6.22302233384499969837122534954688e-01 3.01148582116600005420536945166532e-01 7.65491844989000247423405198787805e-02 2.69252786513499986487119031153270e-02
            3.10378628805099976428749641854665e-01 7.79098365078999960386951784130360e-02 6.11711534687000013654767371917842e-01 2.70947664659500001005376645935030e-02
            8.19218215187000053933985554976971e-02 4.60363303835099990646284595641191e-01 4.57714874646200031715892464490025e-01 2.92368573222000008304810592107970e-02
            4.71702266501299982692785306426231e-01 8.21554006796999941597547945093538e-02 4.46142332819000009269672091249959e-01 2.96431584181500000496622959644810e-02
            4.54660341525000011309032288409071e-01 4.63756503388999985482143983972492e-01 8.15831550860000032088237276184373e-02 2.97179138374500002861555714162023e-02
            1.70109133923700001078671562027012e-01 6.42227780818800053630468482879223e-01 1.87663085257499973046435570722679e-01 3.15900127931499977163021242176910e-02
            6.40600432948699949342596937640337e-01 1.89829353725600002977813574034371e-01 1.69570213325700047679589488325291e-01 3.16463422576499994742782462253672e-02
            1.91226758371700000793680374044925e-01 1.73995568534300010243853762403887e-01 6.34777673094000016718041479180101e-01 3.20353680885999969607524917591945e-02
            1.88531576706999992598667859056150e-01 4.79891407040600015410092282763799e-01 3.31577016252400047502391089437879e-01 4.06020297959000034393461930903868e-02
            4.77292995769100003577989355108002e-01 3.34835659811899999382944770331960e-01 1.87871344418999997039065874560038e-01 4.07218756764999995079179484491760e-02
            3.12697462175999973954532151765306e-01 4.95797219725900017994746349359048e-01 1.91505318098100063561872730133473e-01 4.07339600620499966976417738351302e-02
            4.96122594594600008210960595533834e-01 1.92755366890399987278570392845722e-01 3.11122038515000087777195858507184e-01 4.07525274041999965857208110264764e-02
            1.92880531286699991033728451839124e-01 3.16101580726100006568657363459351e-01 4.91017887987199974642038569072611e-01 4.07582332469499999949569257751136e-02
            3.36004145381600005659805674440577e-01 1.89489280128999987296012363913178e-01 4.74506574489399923777455114759505e-01 4.08465529811500027701498538590386e-02
            3.33728055084800023788460521245725e-01 3.34357102181099985482859437979641e-01 3.31914842734099935217528809516807e-01 4.61609167265499970023157061405072e-02
            ];
        
    case  18
        
        % ALG. DEG.:   18
        % PTS CARD.:   66
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.0e-13
        
        xyw_bar=[
            1.16731059668000007600863199286323e-02 9.81256595128899955504664376348956e-01 7.07029890429999863243892832542770e-03 1.25828784929999997962979385590643e-03
            9.81003085838800026330375203542644e-01 7.14625048630000009319829246123845e-03 1.18506636748999735764265039961174e-02 1.26367260034999993704996068544233e-03
            1.06966317092000006566676617580924e-02 1.15153933376000005422978844649151e-02 9.77787974953200023087163117452292e-01 1.66346476664999990899285187140322e-03
            9.38247698355100023803743169992231e-01 4.95570591341000024621088471121766e-02 1.21952425107999737341479828955926e-02 4.07517460624999983825889060540248e-03
            1.26627518416999994971261500609216e-02 9.37012362061499959331456466316013e-01 5.03248860968000810700573310896289e-02 4.30677628710000003076086017017587e-03
            5.98109409984000015003680061909108e-02 1.21364578921999994448599480278972e-02 9.28052601109400021606177233479684e-01 4.38933730895000023125573918036935e-03
            1.37363297926999996384500590806965e-02 6.12783625596999975693179862901161e-02 9.24985307647600007996402382559609e-01 4.85497927810000021175973472509213e-03
            9.22952795940499948201818369852845e-01 1.41128270601999991629149633354245e-02 6.29343769993000456963727629045025e-02 5.12331059575000039019165853915183e-03
            6.33107354992999971088352140213829e-02 9.22019729172700031050169400259620e-01 1.46695353279999718409953857189976e-02 5.41988441705000024983807094258736e-03
            1.17265100335000004017071972839403e-02 1.50052047522899995035672304766194e-01 8.38221442443599973337597930367338e-01 6.46926950880000010968284840373599e-03
            1.55472058732299994732173331613012e-01 8.32514712158900049843168744700961e-01 1.20132291087999831802335393149406e-02 6.81699117915000006584502756368238e-03
            8.34329388898199963975343962374609e-01 1.25228158758999997407501680868336e-02 1.53147795225900024140841537700908e-01 6.92386640734999996016352241667846e-03
            8.50163803195699951942287952988409e-01 1.37199750873600007095021169334359e-01 1.26364459307000409626908776772325e-02 6.97107700525000009011966639604907e-03
            1.28816350522000001854738826523317e-02 8.47762706347899963610359463928035e-01 1.39355658599900067429189221002162e-01 7.20606999839999966955828014647523e-03
            1.51080160895900011963632891820453e-01 1.36526924038999995097976025704156e-02 8.35267146700199969444611269864254e-01 7.68517277669999998579442035406828e-03
            1.01917879217000000718895691420585e-02 5.77043861834499960394850859302096e-01 4.12764350243800093309687326836865e-01 8.12449011265000027581795194464576e-03
            2.81337239930300020418485473783221e-01 7.06685375962299966445812060555909e-01 1.19773841074000131357024656608701e-02 8.48591521400000013686248934163814e-03
            7.12437462850100011735321459127590e-01 1.24569780989999996656525382832115e-02 2.75105559050900005946260762357269e-01 8.50442662105000031502743951250523e-03
            2.76302525086299988199556310064509e-01 1.21741311386000007621621676889845e-02 7.11523343775099958996577242942294e-01 8.54767603374999990206717370710976e-03
            1.09658368560999995577898147303131e-02 4.19430671246599995338755206830683e-01 5.69603491897300084900734873372130e-01 8.69444272795000076825555623827313e-03
            4.28911051788399999562528819296858e-01 5.59961606746899986930543491325807e-01 1.11273414646999579957764581195079e-02 8.72719812195000069210415460929653e-03
            4.21542055511499980102740892107249e-01 1.16475994784999998393759668147140e-02 5.66810345010000049548182232683757e-01 8.92033786434999996506345354418954e-03
            5.71125859044400052688672531076008e-01 1.18218313988999994962325601477460e-02 4.17052309556699973835947048428352e-01 8.92234319394999940422419371088836e-03
            5.82686827051099975882664239179576e-01 4.05788958117699993799476487765787e-01 1.15242148312000303178592730546370e-02 8.95231687759999970877000663449508e-03
            1.30567806713000004320557323467256e-02 2.72502375086799975001383700146107e-01 7.14440844241900019362390139576746e-01 9.06298781005000081423261804047797e-03
            1.30760400963999996737374686972544e-02 7.22471252323300006104034309828421e-01 2.64452707580300039325038596871309e-01 9.23924194410000003407912316788497e-03
            7.26343706240699993337273099314189e-01 2.60298401925100009179914195556194e-01 1.33578918341999974828127051296178e-02 9.28967821855000014386050111170334e-03
            6.87230068637000035547401921576238e-02 6.31417277210000044140869590592047e-02 8.68135265415299950397809425339801e-01 1.01608575888500005796055702944614e-02
            8.65230210152900025555311458447250e-01 7.20611837337999938668531285657082e-02 6.27086061132999805778354129870422e-02 1.06885830904499996851786391971473e-02
            6.48599071037000013406625953393814e-02 8.59043354390999991387900536210509e-01 7.60967385052999656380734450067393e-02 1.15958427048999997838185649357001e-02
            1.48349494336200010691229067560926e-01 7.88878835224000041215219880541554e-01 6.27716704397999203379754362686072e-02 1.37213355429500003662690588157602e-02
            6.24359898395999979303638838246115e-02 1.49393549935399994055273964477237e-01 7.88170460225000035769937767327065e-01 1.45150961169999998673363350576437e-02
            7.87136901173499992090398791333428e-01 6.56382042756999961596875436953269e-02 1.47224894550800011749913664971245e-01 1.47261369252500006615802519149838e-02
            5.19104921610000005771645703589456e-02 5.25563569560499987964874435419915e-01 4.22525938278500046152430513757281e-01 1.49718125814500002679352519407985e-02
            1.54312992744399996025705945612572e-01 7.16383926916999980161193661842844e-02 7.74048614563899950447023456945317e-01 1.53513474059500006074774347553102e-02
            2.61784274560300012080205078746076e-01 6.21479485287999983134099579729082e-02 6.76067776910899920217445924208732e-01 1.62631682931499987887491442961618e-02
            7.66725787281300030784336740907747e-01 1.65821155483100002303586961716064e-01 6.74530572355999669120762973761884e-02 1.63942104252999994173922715390290e-02
            2.58210367662700002178155500587309e-01 6.80011976613900004373647334432462e-01 6.17776557233999934481971649802290e-02 1.65617337595999995769613377660789e-02
            6.79065925147000065686953007570992e-02 7.57151543778200042389414647914236e-01 1.74941863707099964919677859143121e-01 1.73083763437500010029523878074542e-02
            5.29357827480399945052624843810918e-01 4.12150384110699985473758033549530e-01 5.84917884089000694736171226395527e-02 1.73540686988000007751065112415745e-02
            6.66036150483999972449211668390490e-02 2.61251308788599978516486999069457e-01 6.72145076163000121383106488792691e-01 1.73686024702000008090863047982566e-02
            5.85675461898999966114942594685999e-02 3.90223611453499985568527108625858e-01 5.51208842356599948431039592833258e-01 1.74264381226999984231262885714386e-02
            6.44535360411000041169771179738746e-02 6.37362655976099978794024991657352e-01 2.98183807982799975455634466925403e-01 1.74300780593000005647130734587336e-02
            6.74813842915100048358567619288806e-01 6.37583342060999980205338033556473e-02 2.61427822878799953620898577355547e-01 1.77735784987500013143435495521771e-02
            3.91460231036899974910170385555830e-01 5.50323809056299961817160237842472e-01 5.82159599068000632726693766016979e-02 1.80091498191499999048659219624824e-02
            6.48770149230699977316305648855632e-01 2.83672836026300001499578229413601e-01 6.75570147430000211841161217307672e-02 1.81463142921499988313716045240653e-02
            3.94649822040800013489558750734432e-01 6.05175522554000019925801723275072e-02 5.44832625703799977578967173030833e-01 1.90948851041500000658146518617286e-02
            5.39013715193299991490505362889962e-01 6.11990176935999982688052512003196e-02 3.99787267113100031057371097631403e-01 1.96126400058999998676778631079287e-02
            1.62789508278500000271193925982516e-01 6.86132214103500004398483724799007e-01 1.51078277617999967574746733589564e-01 2.41355062943999984503840039451461e-02
            6.81243632264100007489560084650293e-01 1.56796834589900013767760356131475e-01 1.61959533145999978742679559218232e-01 2.44956060783000015290689077573916e-02
            1.54283287802000013133607581039541e-01 1.66751262402000000051316419558134e-01 6.78965449796000042326227230660152e-01 2.48610416936000007404050649029159e-02
            2.52272775044500019436810589468223e-01 2.50480393339500007066078524076147e-01 4.97246831616000029008262117713457e-01 2.53532868493000010379212483258016e-02
            2.54798153240699987165385209664237e-01 4.99409064904299981879631786796381e-01 2.45792781855000030954983003539382e-01 2.54885997021499985382142483558710e-02
            1.48558054919399995519313506520120e-01 5.75602309608699957976796213188209e-01 2.75839635471900046503890280291671e-01 2.60680031833500011617665848007164e-02
            2.93023960643599978848072851178586e-01 5.65689735416200045747814328933600e-01 1.41286303940199919892961588629987e-01 2.61730437462499992973263829298958e-02
            2.80899127231000023119378283809056e-01 1.43792157424800004505982542468701e-01 5.75308715344200027885790404980071e-01 2.62220341775999989109013910137946e-02
            4.82098959297100004839364828512771e-01 2.51855753586499975238410797828692e-01 2.66045287116400019922224373658537e-01 2.63729822411499989498473439653026e-02
            5.64187824544399973980546292295912e-01 1.46296674315299995994266168963804e-01 2.89515501140300002269611923111370e-01 2.64724531863999995584535440684704e-02
            1.30769964434400004504865933085966e-01 4.48957758611700008533063055438106e-01 4.20272276953899959206495395847014e-01 2.71197797250499993138017629235037e-02
            1.47969222194799993053138109644351e-01 3.00117438682899972945961053483188e-01 5.51913339122300006245325221243547e-01 2.71735101709499986266660442879584e-02
            5.63868422294599991317909370991401e-01 2.81377208929800004533205992629519e-01 1.54754368775600004148884636379080e-01 2.73550274319500005826277799769741e-02
            4.36115742879000012344192782620667e-01 4.25205344641999993804404311958933e-01 1.38678912478999993851402905420400e-01 2.78644172956499999727064675880683e-02
            3.60326393528499988594404612740618e-01 2.59919000488899987200142049914575e-01 3.79754605982600079716604568602634e-01 2.88867132116500009708648377682039e-02
            4.22418833467400001158154054792249e-01 1.45323844330299989380250735848676e-01 4.32257322202300064972746440616902e-01 2.92696890811500014251134160758738e-02
            3.71900183305200027206183222006075e-01 3.78012270356699986439963367956807e-01 2.50087546338099986353853410037118e-01 3.04519625339999995550233080621183e-02
            2.41364500692799993641912692510232e-01 3.84756328493999988360485531302402e-01 3.73879170813199990242026160558453e-01 3.18636982224499987204069384461036e-02
            ];
        
    case  20
        
        % ALG. DEG.:   20
        % PTS CARD.:   78
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 7.7e-13
        
        xyw_bar=[
            8.94113371120000034741881478339565e-03 8.69832937020000029892941029174835e-03 9.82360536918600035782844770437805e-01 1.08722726994999992558565971023654e-03
            9.79262262980699960124297831498552e-01 1.02644133744000005947771825276504e-02 1.04733236449000392809249859737974e-02 1.44935676325000004603660297419765e-03
            1.05475382112000005963015070165056e-02 9.78551420251499992630783708591480e-01 1.09010415372999824867861207167152e-02 1.54230146684999989276088072642779e-03
            2.37770619469999987607988778393064e-03 6.36551098604000026304206016902754e-02 9.33967183944899970171604763891082e-01 1.72008165519999998115119321795419e-03
            6.30425115794999990637137443627580e-02 4.15063475090000043310478616831460e-03 9.32806853669599922440625050512608e-01 2.09492360060000012630943011515683e-03
            9.30842249672999999532407855440397e-01 4.80534822629999965742930712053749e-03 6.43524021007000068816950033578905e-02 2.23690257489999981543871143685465e-03
            6.29076555490000050019361310660315e-02 9.31679006948100019158687246090267e-01 5.41333750289996196158881502924487e-03 2.35272104070000011993846200653024e-03
            9.31596224638099990045247977832332e-01 6.26264881800999984040601020751637e-02 5.77728718180001155069192009250401e-03 2.44339678750000012866228082941689e-03
            6.19516894150000038687942094384198e-03 9.29358705856400013622931055579102e-01 6.44461252021000241541059949668124e-02 2.59638216844999998589305079121914e-03
            2.87125819236999993422365662354423e-02 3.10202122996999993254085836724698e-02 9.40267205776600056843506081349915e-01 3.70365294905000003167216959809593e-03
            9.29384447830500004528175850282423e-01 3.42152968218999986582673500379315e-02 3.64002553475999968135567996796453e-02 3.98777051504999990177990909501204e-03
            3.75457566620999994011498301915708e-02 9.25786888466900048655361388227902e-01 3.66673548709999241879131659516133e-02 4.17752614550000005155805382628387e-03
            8.68957390639999924242253825923399e-03 1.58497125151000006759716143278638e-01 8.32813300942599976650626558694057e-01 4.80833304319999970538601630209996e-03
            1.54759705396499996910009144812648e-01 8.36360665768799971253599778719945e-01 8.87962883470005959196669209632091e-03 4.81591289249999957022829732977698e-03
            8.33102529418500048485896058991784e-01 8.92572448239999961805413164483980e-03 1.57971746099099946691879381432955e-01 4.92887303790000026043616188076157e-03
            8.37423107352599949138038937235251e-01 1.52916730407799988933348345199192e-01 9.66016223960006192861271756555652e-03 5.13289401505000013326007746172763e-03
            1.55936250523399988487938117032172e-01 9.49662400579999975425149472130215e-03 8.34567125470800030839768623991404e-01 5.15940515554999968445004654427066e-03
            9.85996420949999952121345359046245e-03 8.34221149359600011585769152588909e-01 1.55918886430900016648593009449542e-01 5.31455008149999980765487705980377e-03
            4.05587373328899991609119979329989e-01 7.43893020079999973959150594282619e-03 5.86973696470300065897163221961819e-01 5.34406534474999965478314578604113e-03
            5.96472789861799967958688739599893e-01 3.95633080931100000743327882446465e-01 7.89412920710003129798337795364205e-03 5.34845105049999974128338564582918e-03
            8.07478004159999963873506345635178e-03 4.03131942590299996442837482391042e-01 5.88793277368099943203105794964358e-01 5.45132308570000015390810688131751e-03
            7.50739777210000010926327362881239e-03 5.85160959468099983737943148298655e-01 4.07331642759800049979901359620271e-01 5.49498917874999964761073556474003e-03
            3.93676451923699988189042642261484e-01 5.97489659289900054695010567229474e-01 8.83388878640001262709802176686935e-03 5.67115276144999989199613210644202e-03
            5.84653072621200031733224022900686e-01 8.72504649680000063383111807979731e-03 4.06621880881999964163497907065903e-01 6.02678214650000012436725782549729e-03
            4.87080411212000019549606122382102e-01 2.02129229911999983015924442497635e-02 4.92706665796800058476634376347647e-01 6.98095969105000033194663444646721e-03
            2.68351281178499989632513234028011e-01 7.20234008866800001769092887116130e-01 1.14147099546999530872426475980319e-02 7.05739957679999991430053185581528e-03
            7.22395628874800022067859117669286e-01 2.66239936645599972742104455392109e-01 1.13644344796000051900364269386046e-02 7.09651735230000003940009634106900e-03
            2.71682674235699972431490323288017e-01 1.12882698808000007245633256047768e-02 7.17029055883500077150927154434612e-01 7.21063381340000027902581436478613e-03
            1.12580842045999999362360100008118e-02 7.16969596332500014312927305581979e-01 2.71772319462900036057817487744614e-01 7.23521734274999977348796775800110e-03
            1.15034734370000005926337749428967e-02 2.74006711016600013053334805590566e-01 7.14489815546400008905436607165029e-01 7.24748849360000012981286587887553e-03
            7.14052590056399982820778404857265e-01 1.13511560497000004582579535394871e-02 2.74596253893900021925134069533669e-01 7.26933878469999961158265477934037e-03
            4.90287105311200011037442436645506e-01 4.93649184146799979711062178466818e-01 1.60637105420000092514953848876758e-02 7.29820954629999997520517496241155e-03
            2.01423425209000007962689693385983e-02 4.83257345960100004322868016970460e-01 4.96600311519000026105885581273469e-01 7.36572892330000002919510038168482e-03
            3.61107464858999999135491520974028e-02 9.35679501582000056059129633467819e-02 8.70321303355900077747264731442556e-01 8.37319816519999961246245590018589e-03
            8.60799881985100046044578903092770e-01 3.97379067074999994813033765694854e-02 9.94622113073999614130116242449731e-02 8.44777502290000072149300791579662e-03
            1.00589152600099998591254291113728e-01 8.58634341935200051665333376149647e-01 4.07765054646999081100489092932548e-02 8.47113314419999964555163529666970e-03
            9.18740717058000033734543876562384e-02 3.95513001973000002942093544788804e-02 8.68574628096900003271230161772110e-01 8.65350860475000080440644012469420e-03
            8.60488829619100004109100154892076e-01 9.66224057078999970515909012647171e-02 4.28887646729999988393089438432071e-02 8.72622732464999918688963731483454e-03
            4.39842178673000017141703210654669e-02 8.56188634910699963853630833909847e-01 9.98271472220000344321988450246863e-02 8.86086110795000075901484137830266e-03
            2.01101760673499996290303215573658e-01 7.44911583562600010388621285528643e-01 5.39866557639000488322267301555257e-02 1.41412012011500002467423442453764e-02
            7.44999372626300004895938400295563e-01 5.36865638166000025410795615243842e-02 2.01314063557099992562982038180053e-01 1.42498356243999993453464014692145e-02
            5.32186641310000013715963973481848e-02 1.96375427593499990441827662834839e-01 7.50405908275500022064363747631432e-01 1.42502823269499995922915402957187e-02
            7.45398464740099964132014065398835e-01 1.98206580555000011489852340673679e-01 5.63949547049000243781335939274868e-02 1.50323611738999995957044575334294e-02
            1.95728993287599989381320142456389e-01 5.55713833155999994972695787964767e-02 7.48699623396800073571455413912190e-01 1.51015638540999991984437045289269e-02
            1.09253205798800004378001915483765e-01 6.10003618241300005209382106841076e-01 2.80743175959899948779252554231789e-01 1.51993568038499993894108186509584e-02
            5.67625702000999973750872129585332e-02 7.40912189495899986546589843783295e-01 2.02325240304000009139429039350944e-01 1.52834398037000000064589144699312e-02
            4.83837933475000023242351687713381e-02 6.07513566097800050691546402958920e-01 3.44102640554699967800900139991427e-01 1.53033706501000000416157220683999e-02
            1.08061280975999998710257443690352e-01 1.12208151043699994042768253166287e-01 7.79730567980300048880337726586731e-01 1.54665034100500003999556497547019e-02
            6.18560590099100027217104980081785e-01 2.69875370303500006219366014192929e-01 1.11564039597399966563529005725286e-01 1.54886910417499999020884970946099e-02
            7.72129601349699989931707477808231e-01 1.11411739533299997129134339957091e-01 1.16458659117000012939158182234678e-01 1.56573125272500002225228854513261e-02
            6.11573480113300038674140068906127e-01 3.38936767793100002243988910777261e-01 4.94897520935999590818710203166120e-02 1.56786746696000005296678381228048e-02
            3.38132610337600003624913824751275e-01 4.94693938787000020429474034244777e-02 6.12397995783700022087714387453161e-01 1.57160234643499985285330922124558e-02
            1.17308412825399999146469554034411e-01 7.69645130979500047274655116780195e-01 1.13046456195099898067724097927567e-01 1.57591071947000005426531288321712e-02
            2.67455126059599979093661659135250e-01 1.11571880815399995801051602484222e-01 6.20972993124999983471923314937158e-01 1.62124068992499995034428650342306e-02
            6.54210016002599981099763226666255e-01 1.90654831470000007964671340232599e-01 1.55135152527400010935565433101146e-01 1.73756076193000014162848287924135e-02
            5.38297481157999982426254348411021e-02 3.35861682684900020312568358349381e-01 6.10308569199299988383700110716745e-01 1.75196727463499991872541983184419e-02
            1.84884032411700011122590581180702e-01 1.55183152385099998316064784376067e-01 6.59932815203200018316920250072144e-01 1.75358710154999986041701731664944e-02
            3.37626710474400015815632514204481e-01 6.08140259629399992924447815312305e-01 5.42330298962000467710709017410409e-02 1.76064607666999994872725210370845e-02
            6.06710203449900009609052631276427e-01 5.42632795598000025605323060062801e-02 3.39026516990300008647096774438978e-01 1.76307752490500006803397781141030e-02
            4.61261408549600016026914772737655e-01 6.88176670721999950153247027628822e-02 4.69920924378199933446609293241636e-01 1.83201610171500016421841650071656e-02
            1.52546536567099988435103341544163e-01 6.51024084574900019362075909157284e-01 1.96429378858000047713971980556380e-01 1.83866553835000008654265712948472e-02
            7.00582543543000063612780081712117e-02 4.66190439274200008057391642068978e-01 4.63751306371499971703542541945353e-01 1.85837831468500004616917209432359e-02
            4.70420137903199975237811258921283e-01 4.63482645535299997430200846793014e-01 6.60972165614999718208366630278761e-02 1.86685785803000006688456835490797e-02
            1.21646169374600002233144380170415e-01 2.38149487551600008705676714271249e-01 6.40204343073799919672239866486052e-01 2.01986673294000003420212152605018e-02
            6.37140405270199994269830767734675e-01 1.23839938451299994004450866214029e-01 2.39019656278500025603506173865753e-01 2.06790020318999984150210025291017e-02
            2.37990451511899986369868997826416e-01 6.37021645232599964536746028898051e-01 1.24987903255500021337809357646620e-01 2.10978895935000007599846583161707e-02
            1.48392985717699987802120631386060e-01 4.89418857777999982694439040642465e-01 3.62188156504300029503440327971475e-01 2.47725502018499993406308590238041e-02
            3.59806957154999995651678545982577e-01 1.45288086625299989185222671039810e-01 4.94904956219699987407523167348700e-01 2.50209630570500003721168269521513e-02
            4.94144105509499997896938339181361e-01 3.61021638381800014094835660216631e-01 1.44834256108700043519377231859835e-01 2.52897293557499992788795140086222e-02
            1.44063068798099991330019520319183e-01 3.51350834188700000826344194138073e-01 5.04586097013200118865938748058397e-01 2.60018605093999995159226301666422e-02
            5.01976444000399979472604172769934e-01 1.43549166329300009747882427291188e-01 3.54474389670299983023937784309965e-01 2.60766783943000013401558589976048e-02
            3.55542383429799990768316320099984e-01 5.01649159950199985402718994009774e-01 1.42808456620000079340115917148069e-01 2.62449576179000013120834466917586e-02
            2.44343954077099995148003586109553e-01 2.40605212910399990366983047351823e-01 5.15050833012500097751740213425364e-01 2.99579881258000002564045871622511e-02
            2.43706498934199988770643585667131e-01 5.10901727705500019460771454760106e-01 2.45391773360300047279736190830590e-01 2.99804998713000016297147709565252e-02
            5.12220080732100013776175728708040e-01 2.45273797354299993855875072767958e-01 2.42506121913599992367949198524002e-01 2.99957636064499995331988912994348e-02
            2.52603831517800025974196387323900e-01 3.70031955509399990145169567767880e-01 3.77364212972799928369482813650393e-01 3.17066591724499968751871392669273e-02
            3.75989565285100024727427125981194e-01 2.50540661163099975805579333609785e-01 3.73469773551800054978144771666848e-01 3.17655930553999987786539804801578e-02
            3.72907798714399996153900929130032e-01 3.75375027754900003618132586780121e-01 2.51717173530699944716815252832021e-01 3.18603302835999968589675290786545e-02
            ];
        
    case  21
        
        % ALG. DEG.:   21
        % PTS CARD.:   91
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 8.9e-13
        
        xyw_bar=[
            3.55243919219999995662284675290721e-03 3.55243919219999995662284675290721e-03 9.92895121615599896003345747885760e-01 3.35221821950000000659569687755379e-04
            3.55243919219999995662284675290721e-03 9.92895121615600007025648210401414e-01 3.55243919219994097602466354146600e-03 3.35221821950000000659569687755379e-04
            9.92895121615600007025648210401414e-01 3.55243919219999995662284675290721e-03 3.55243919219999301772894284567883e-03 3.35221821950000000659569687755379e-04
            9.55354827373000037304962006601272e-01 8.78989290930000041757175210932473e-03 3.58552797176999588080192893357889e-02 2.27363040370000008688622372687860e-03
            3.58552797177000004413827127791592e-02 8.78989290930000041757175210932473e-03 9.55354827373000037304962006601272e-01 2.27363040370000008688622372687860e-03
            9.55354827373000037304962006601272e-01 3.58552797177000004413827127791592e-02 8.78989290929996225365528061956866e-03 2.27363040370000008688622372687860e-03
            8.78989290930000041757175210932473e-03 3.58552797177000004413827127791592e-02 9.55354827373000037304962006601272e-01 2.27363040370000008688622372687860e-03
            8.78989290930000041757175210932473e-03 9.55354827373000037304962006601272e-01 3.58552797176999726858070971502457e-02 2.27363040370000008688622372687860e-03
            3.58552797177000004413827127791592e-02 9.55354827373000037304962006601272e-01 8.78989290929999000923089624848217e-03 2.27363040370000008688622372687860e-03
            8.86526487904700011277725479885703e-01 1.08232974501699996672243742068531e-01 5.24053759359999205003077804576606e-03 2.60387926600000015400770969620226e-03
            8.86526487904700011277725479885703e-01 5.24053759349999998679026091963351e-03 1.08232974501799986133399045229453e-01 2.60387926600000015400770969620226e-03
            5.24053759349999998679026091963351e-03 1.08232974501699996672243742068531e-01 8.86526487904800042372244206489995e-01 2.60387926600000015400770969620226e-03
            5.24053759349999998679026091963351e-03 8.86526487904700011277725479885703e-01 1.08232974501800027766762468672823e-01 2.60387926600000015400770969620226e-03
            1.08232974501699996672243742068531e-01 8.86526487904700011277725479885703e-01 5.24053759359999205003077804576606e-03 2.60387926600000015400770969620226e-03
            1.08232974501699996672243742068531e-01 5.24053759349999998679026091963351e-03 8.86526487904800042372244206489995e-01 2.60387926600000015400770969620226e-03
            4.66397432150000013084678585073561e-02 9.06720513570000052894215514243115e-01 4.66397432149999735528922428784426e-02 3.27177164435000014663934564396186e-03
            4.66397432150000013084678585073561e-02 4.66397432150000013084678585073561e-02 9.06720513570000052894215514243115e-01 3.27177164435000014663934564396186e-03
            9.06720513570000052894215514243115e-01 4.66397432150000013084678585073561e-02 4.66397432149999457973166272495291e-02 3.27177164435000014663934564396186e-03
            2.07572045694599999210439023045183e-01 8.27592412839999944096369688395498e-03 7.84152030177000081145877175003989e-01 4.63689207664999981234332793178510e-03
            2.07572045694599999210439023045183e-01 7.84152030176999970123574712488335e-01 8.27592412840005842156188009539619e-03 4.63689207664999981234332793178510e-03
            7.84152030176999970123574712488335e-01 2.07572045694599999210439023045183e-01 8.27592412840003066598626446648268e-03 4.63689207664999981234332793178510e-03
            8.27592412839999944096369688395498e-03 7.84152030176999970123574712488335e-01 2.07572045694600082477165869931923e-01 4.63689207664999981234332793178510e-03
            8.27592412839999944096369688395498e-03 2.07572045694599999210439023045183e-01 7.84152030177000081145877175003989e-01 4.63689207664999981234332793178510e-03
            7.84152030176999970123574712488335e-01 8.27592412839999944096369688395498e-03 2.07572045694600026966014638674096e-01 4.63689207664999981234332793178510e-03
            8.58119489724999962021456667571329e-02 3.14836947700999972288293804467685e-02 8.82704356257399957996767625445500e-01 4.79688913115000007542088056311513e-03
            8.82704356257399957996767625445500e-01 3.14836947700999972288293804467685e-02 8.58119489725000517132968980149599e-02 4.79688913115000007542088056311513e-03
            3.14836947700999972288293804467685e-02 8.58119489724999962021456667571329e-02 8.82704356257399957996767625445500e-01 4.79688913115000007542088056311513e-03
            8.58119489724999962021456667571329e-02 8.82704356257399957996767625445500e-01 3.14836947701000458010867077973671e-02 4.79688913115000007542088056311513e-03
            8.82704356257399957996767625445500e-01 8.58119489724999962021456667571329e-02 3.14836947701000458010867077973671e-02 4.79688913115000007542088056311513e-03
            3.14836947700999972288293804467685e-02 8.82704356257399957996767625445500e-01 8.58119489724999962021456667571329e-02 4.79688913115000007542088056311513e-03
            6.68877823382600045221124673844315e-01 9.51507606250000075032158264320969e-03 3.21607100554899938416042459721211e-01 5.71239045834999979722734764209235e-03
            9.51507606250000075032158264320969e-03 3.21607100555000025021712417583331e-01 6.68877823382499903104303484724369e-01 5.71239045834999979722734764209235e-03
            9.51507606250000075032158264320969e-03 6.68877823382600045221124673844315e-01 3.21607100554899938416042459721211e-01 5.71239045834999979722734764209235e-03
            6.68877823382600045221124673844315e-01 3.21607100555000025021712417583331e-01 9.51507606239992975716290857235435e-03 5.71239045834999979722734764209235e-03
            3.21607100555000025021712417583331e-01 6.68877823382600045221124673844315e-01 9.51507606239987424601167731452733e-03 5.71239045834999979722734764209235e-03
            3.21607100555000025021712417583331e-01 9.51507606250000075032158264320969e-03 6.68877823382499903104303484724369e-01 5.71239045834999979722734764209235e-03
            4.37999954311300010445506813994143e-01 9.98597856810000059080145717871346e-03 5.52014067120599971616456969059072e-01 5.86084820870000008929956436531938e-03
            9.98597856810000059080145717871346e-03 5.52014067120599971616456969059072e-01 4.37999954311300010445506813994143e-01 5.86084820870000008929956436531938e-03
            4.37999954311300010445506813994143e-01 5.52014067120599971616456969059072e-01 9.98597856810001793803621694678441e-03 5.86084820870000008929956436531938e-03
            9.98597856810000059080145717871346e-03 4.37999954311300010445506813994143e-01 5.52014067120599971616456969059072e-01 5.86084820870000008929956436531938e-03
            5.52014067120599971616456969059072e-01 4.37999954311300010445506813994143e-01 9.98597856810001793803621694678441e-03 5.86084820870000008929956436531938e-03
            5.52014067120599971616456969059072e-01 9.98597856810000059080145717871346e-03 4.37999954311300010445506813994143e-01 5.86084820870000008929956436531938e-03
            7.97493107214800045490221691579791e-01 4.05093994119000017084530895772332e-02 1.61997493373299938923537411028519e-01 9.40985776160000071677291799687737e-03
            4.05093994119000017084530895772332e-02 1.61997493373399997773631753261725e-01 7.97493107214700014395702964975499e-01 9.40985776160000071677291799687737e-03
            4.05093994119000017084530895772332e-02 7.97493107214800045490221691579791e-01 1.61997493373299938923537411028519e-01 9.40985776160000071677291799687737e-03
            1.61997493373399997773631753261725e-01 7.97493107214800045490221691579791e-01 4.05093994117999844917221707873978e-02 9.40985776160000071677291799687737e-03
            7.97493107214800045490221691579791e-01 1.61997493373399997773631753261725e-01 4.05093994117999567361465551584843e-02 9.40985776160000071677291799687737e-03
            1.61997493373399997773631753261725e-01 4.05093994119000017084530895772332e-02 7.97493107214700014395702964975499e-01 9.40985776160000071677291799687737e-03
            3.86421555195499977219952825180371e-01 3.86421555195499977219952825180371e-01 2.27156889609000101071245580897084e-01 1.17630490135500007792268561956917e-02
            3.86421555195499977219952825180371e-01 2.27156889608999990048943118381430e-01 3.86421555195500088242255287696025e-01 1.17630490135500007792268561956917e-02
            2.27156889608999990048943118381430e-01 3.86421555195499977219952825180371e-01 3.86421555195500088242255287696025e-01 1.17630490135500007792268561956917e-02
            8.09012937932899967030664356570924e-01 9.54935310336000042763515693877707e-02 9.54935310335000286929840740413056e-02 1.17785733075499998712132310174638e-02
            9.54935310336000042763515693877707e-02 8.09012937932899967030664356570924e-01 9.54935310335000009374084584123921e-02 1.17785733075499998712132310174638e-02
            9.54935310336000042763515693877707e-02 9.54935310336000042763515693877707e-02 8.09012937932799935936145629966632e-01 1.17785733075499998712132310174638e-02
            2.74542523871800003831822323263623e-01 4.79840480721000009478593995027040e-02 6.77473428056100002159212181140902e-01 1.34123103714999993701928460154704e-02
            4.79840480721000009478593995027040e-02 6.77473428056100002159212181140902e-01 2.74542523871800003831822323263623e-01 1.34123103714999993701928460154704e-02
            6.77473428056100002159212181140902e-01 4.79840480721000009478593995027040e-02 2.74542523871800003831822323263623e-01 1.34123103714999993701928460154704e-02
            6.77473428056100002159212181140902e-01 2.74542523871800003831822323263623e-01 4.79840480720999940089654955954757e-02 1.34123103714999993701928460154704e-02
            2.74542523871800003831822323263623e-01 6.77473428056100002159212181140902e-01 4.79840480720999940089654955954757e-02 1.34123103714999993701928460154704e-02
            4.79840480721000009478593995027040e-02 2.74542523871800003831822323263623e-01 6.77473428056100002159212181140902e-01 1.34123103714999993701928460154704e-02
            4.05347244666700023962846444192110e-01 5.42984962234399959335462426679442e-01 5.16677930989000167016911291284487e-02 1.57144888389499984282515754330234e-02
            5.16677930989000028239033213139919e-02 4.05347244666700023962846444192110e-01 5.42984962234399959335462426679442e-01 1.57144888389499984282515754330234e-02
            4.05347244666700023962846444192110e-01 5.16677930989000028239033213139919e-02 5.42984962234399959335462426679442e-01 1.57144888389499984282515754330234e-02
            5.42984962234399959335462426679442e-01 5.16677930989000028239033213139919e-02 4.05347244666700023962846444192110e-01 1.57144888389499984282515754330234e-02
            5.16677930989000028239033213139919e-02 5.42984962234399959335462426679442e-01 4.05347244666700023962846444192110e-01 1.57144888389499984282515754330234e-02
            5.42984962234399959335462426679442e-01 4.05347244666700023962846444192110e-01 5.16677930989000167016911291284487e-02 1.57144888389499984282515754330234e-02
            1.87773861553899995557870283846569e-01 1.06814826758799996664350828723400e-01 7.05411311687299980022203271801118e-01 1.68598096079499992772099403737229e-02
            7.05411311687199948927684545196826e-01 1.87773861553899995557870283846569e-01 1.06814826758900055514445170956606e-01 1.68598096079499992772099403737229e-02
            7.05411311687199948927684545196826e-01 1.06814826758799996664350828723400e-01 1.87773861554000054407964626079774e-01 1.68598096079499992772099403737229e-02
            1.06814826758799996664350828723400e-01 7.05411311687199948927684545196826e-01 1.87773861553999998896813394821947e-01 1.68598096079499992772099403737229e-02
            1.87773861553899995557870283846569e-01 7.05411311687199948927684545196826e-01 1.06814826758900083270020786585519e-01 1.68598096079499992772099403737229e-02
            1.06814826758799996664350828723400e-01 1.87773861553899995557870283846569e-01 7.05411311687299980022203271801118e-01 1.68598096079499992772099403737229e-02
            1.19505971200899993589494840762200e-01 3.05712299064299974915570601297077e-01 5.74781729734800017617146750126267e-01 2.13872647106500002323059561604168e-02
            1.19505971200899993589494840762200e-01 5.74781729734800017617146750126267e-01 3.05712299064299974915570601297077e-01 2.13872647106500002323059561604168e-02
            5.74781729734800017617146750126267e-01 1.19505971200899993589494840762200e-01 3.05712299064299974915570601297077e-01 2.13872647106500002323059561604168e-02
            5.74781729734800017617146750126267e-01 3.05712299064299974915570601297077e-01 1.19505971200900007467282648576656e-01 2.13872647106500002323059561604168e-02
            3.05712299064299974915570601297077e-01 5.74781729734800017617146750126267e-01 1.19505971200900007467282648576656e-01 2.13872647106500002323059561604168e-02
            3.05712299064299974915570601297077e-01 1.19505971200899993589494840762200e-01 5.74781729734800017617146750126267e-01 2.13872647106500002323059561604168e-02
            5.98124574336299952292961279454175e-01 2.00937712831900011645203107946145e-01 2.00937712831800036061835612599680e-01 2.20569466368500007036335119892101e-02
            2.00937712831900011645203107946145e-01 5.98124574336299952292961279454175e-01 2.00937712831800063817411228228593e-01 2.20569466368500007036335119892101e-02
            2.00937712831900011645203107946145e-01 2.00937712831900011645203107946145e-01 5.98124574336200032220745015365537e-01 2.20569466368500007036335119892101e-02
            2.16077520000500011843058700833353e-01 3.12136025667299976671387184978812e-01 4.71786454332199955974402882930008e-01 2.30734797342000003128603680124797e-02
            3.12136025667299976671387184978812e-01 2.16077520000500011843058700833353e-01 4.71786454332199955974402882930008e-01 2.30734797342000003128603680124797e-02
            2.16077520000500011843058700833353e-01 4.71786454332099980391035387583543e-01 3.12136025667399952254754680325277e-01 2.30734797342000003128603680124797e-02
            3.12136025667299976671387184978812e-01 4.71786454332099980391035387583543e-01 2.16077520000599987426426196179818e-01 2.30734797342000003128603680124797e-02
            4.71786454332099980391035387583543e-01 3.12136025667299976671387184978812e-01 2.16077520000600042937577427437645e-01 2.30734797342000003128603680124797e-02
            4.71786454332099980391035387583543e-01 2.16077520000500011843058700833353e-01 3.12136025667400007765905911583104e-01 2.30734797342000003128603680124797e-02
            4.37657990384899997504675184245571e-01 4.37657990384899997504675184245571e-01 1.24684019230199949479498400251032e-01 2.34576234311999996806630264245541e-02
            4.37657990384899997504675184245571e-01 1.24684019230299994451804934669781e-01 4.37657990384799966410156457641278e-01 2.34576234311999996806630264245541e-02
            1.24684019230299994451804934669781e-01 4.37657990384899997504675184245571e-01 4.37657990384799966410156457641278e-01 2.34576234311999996806630264245541e-02
            3.33333333333300008138877501551178e-01 3.33333333333300008138877501551178e-01 3.33333333333400039233396228155470e-01 2.75599990173500015988583555781588e-02
            ];
        
    case  23
        
        % ALG. DEG.:   23
        % PTS CARD.:  105
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 7.0e-13
        
        xyw_bar=[
            8.78093038360000001374405798060252e-03 9.90367643677199982477077355724759e-01 8.51425939200067816159389622043818e-04 3.21914913049999994610478326606540e-04
            9.90367531421999958141100250941236e-01 8.78092162320000084452242816723810e-03 8.51546954800041014377320891526324e-04 3.21920653799999986581503730320719e-04
            2.70292764499999995861378998540658e-03 3.35914404439000002788873189274454e-02 9.63705631911099924735708555090241e-01 5.06736785500000040845247628595871e-04
            3.35909214524000021562244455708424e-02 2.70289467099999987723113470394765e-03 9.63706183876600008808566144580254e-01 5.06737628799999967159350955370201e-04
            9.16750686059999986277091466035927e-03 9.16763530509999945150578781749573e-03 9.81664857834300019767681533267023e-01 9.83996496750000077674558340845579e-04
            9.67556818255799977457343175046844e-01 8.47371766560000086454529366619681e-03 2.39694640786000216781115312869588e-02 1.67336568920000003032888535869915e-03
            8.47372006879999924155999480035462e-03 9.67556943534499946935056868824176e-01 2.39693363967000694358944201667327e-02 1.67336696039999989482627285042327e-03
            7.87819487920000081915272716059917e-03 6.76784943861999943592877571063582e-02 9.24443310734599954514578712405637e-01 2.14366616874999988545669182826714e-03
            6.76785477699999998835167502875265e-02 7.87816592909999953431299246631170e-03 9.24443286300900002316893733222969e-01 2.14367299425000015083986326658305e-03
            9.47026695504699977945506361720618e-01 4.42974541187000012132379822560324e-02 8.67585037660002084125565602334973e-03 2.15019006859999998668153686764981e-03
            4.42974755679999970814364473881142e-02 9.47026667648700026802544016391039e-01 8.67585678329996223823172840639018e-03 2.15019245490000003584452059612886e-03
            9.14424321488199964846899092663080e-01 8.17354551320000066727544663081062e-03 7.74021329986000344858254607061099e-02 2.84673146024999991834936174939230e-03
            8.17354244589999986980721757845458e-03 9.14424423403100039564606049680151e-01 7.74020341509999987295032042311504e-02 2.84673200670000018513983164325509e-03
            2.49745229274100005723013850911229e-01 3.83323243471999974829600432713050e-01 3.66931527253899991691810100746807e-01 3.08219340075000002368565965582548e-03
            3.83323264605499980373082280493691e-01 2.49745126800500011343686423970212e-01 3.66931608593999980527655679907184e-01 3.08223782090000014352582091703425e-03
            8.87685035355699980641475121956319e-01 1.03532880944600003125799503322924e-01 8.78208369970001623272537472075783e-03 3.10072567954999996076015378321244e-03
            1.03532922829700002598229957584408e-01 8.87684993183999959676100388605846e-01 8.78208398630009323682088506757282e-03 3.10072659760000013048908407142790e-03
            7.72559236180000042520932268530487e-03 1.40319099197399987932755038855248e-01 8.51955308440800029856632136215921e-01 3.48181651470000005921257013596914e-03
            1.40319242510699992987355244622449e-01 7.72559346239999958361188703293010e-03 8.51955164026899969265116396854864e-01 3.48181659210000009688212863068202e-03
            8.10459100965200018862333308788948e-01 1.80964252392600000174383012563339e-01 8.57664664219998096328367864771280e-03 3.75331288599999989691324842056019e-03
            1.80964300371699998715158130835334e-01 8.10459051533400032241161170531996e-01 8.57664809489999679925631426158361e-03 3.75331322824999993881256443728489e-03
            8.33076794868400027915811278944602e-01 8.30109396770000070031603200959580e-03 1.58622111163899959240808357208152e-01 3.95373841694999974410995591256324e-03
            8.30109071260000039305104735376517e-03 8.33076854539199995741682869265787e-01 1.58622054748200014273606939241290e-01 3.95373862424999982650986041221586e-03
            3.48407706146999993079838020548777e-02 3.48406969482000020144063512361754e-02 9.30318532437099943166458615451120e-01 4.01766723115000012195308087825651e-03
            2.74028767960800001546317616885062e-01 7.17398184794799997909819921915187e-01 8.57304724440000054386246119975112e-03 4.39817205369999984776452350843101e-03
            7.17398222477800051599672315205680e-01 2.74028730438599987806469471252058e-01 8.57304708359996059385821354226209e-03 4.39817240560000030646925495148025e-03
            2.39497685823400002780303452709632e-01 8.18591822620000006405227566119720e-03 7.52316395950400007563985127490014e-01 4.56520978580000034996455582358976e-03
            8.18591858450000084412856438120798e-03 2.39497556667700006727272921125405e-01 7.52316524747799975081363754725317e-01 4.56521068055000013574673545235783e-03
            6.88361520749999986762412973462233e-03 4.84374089268700014354607219502213e-01 5.08742295523799903378403541864827e-01 4.64108743755000004160837789868310e-03
            4.84374148569900020433465215319302e-01 6.88362329490000042497843679711877e-03 5.08742228135199980876279823860386e-01 4.64109078310000026934556416335909e-03
            4.96076777274099989245570441198652e-01 4.96076752950699972899428757955320e-01 7.84646977520009336615203210385516e-03 4.72499030890000028953812361010023e-03
            6.11293677624500042533384203125024e-01 3.80432369123899993823556542338338e-01 8.27395325159996364305925453663804e-03 4.73137342420000030218663056302830e-03
            3.80432398034500018901837847806746e-01 6.11293646653299993509733667451655e-01 8.27395531219998758842848474159837e-03 4.73137426469999976264002583548063e-03
            7.30389071352400054415454633272020e-01 8.39871797010000031846743695496116e-03 2.61212210677499945266077929773019e-01 4.77778861424999964913062910909503e-03
            8.39871686390000066202432549289369e-03 7.30389089540700031655262591812061e-01 2.61212193595399932988243563158903e-01 4.77778964215000000875788188636761e-03
            6.12852567561199990109344071242958e-01 7.54759796949999988563417829823265e-03 3.79599834469300012607106964424020e-01 4.80694212439999966207659909400718e-03
            7.54759610370000014190061676799814e-03 6.12852548458200030445652828348102e-01 3.79599855438099931248530083394144e-01 4.80694234130000022653383950910211e-03
            7.95253165130000057303494287452850e-03 3.55977382672099995630787816480733e-01 6.36070085676599950019749485363718e-01 4.99957621059999961221631537000576e-03
            3.55977487046000018455771396475029e-01 7.95253585020000083871671847646212e-03 6.36069977103800088258367395610549e-01 4.99957759250000029205285301259210e-03
            9.11023697796599973841580322186928e-01 4.37233665344999994362140682824247e-02 4.52529356689000267222056095306471e-02 5.01506596384999978749386073673122e-03
            4.37233605166000016684080264894874e-02 9.11023680744600050651627043407643e-01 4.52529587387999754355405457317829e-02 5.01506733179999959065797909829598e-03
            3.88480061834999973457982491709117e-02 9.67030908282000001285894086322514e-02 8.64448902988300016403400150011294e-01 6.24683380924999972355626809417117e-03
            9.67032117936000024105425154630211e-02 3.88479942385999976739441308382084e-02 8.64448793967800055426664584956598e-01 6.24683630624999972069666398510890e-03
            8.73226911312000031450963888346450e-02 8.73226620390999963561284857860301e-02 8.25354646829700056009926356637152e-01 7.00986545685000021072008635769635e-03
            4.21445202084000034026267655917763e-02 8.48561778910799979769308265531436e-01 1.09293700880800037644746680598473e-01 7.16681084479999994107934924159053e-03
            8.48561797496099945448122525704093e-01 4.21445420915000021344631875308551e-02 1.09293660412400045478520382857823e-01 7.16681360625000003078399757328043e-03
            8.47792133386400026395790519018192e-01 1.06743594247199993785280014435557e-01 4.54642723663999798189294665462512e-02 7.68020713700000041274096318488773e-03
            1.06743588939800002068913897801394e-01 8.47792132814599974643954283237690e-01 4.54642782456000649204952424042858e-02 7.68020917124999976527410794346906e-03
            1.83396652199100007596399564135936e-01 4.16340521608000030395757562473591e-02 7.74969295640100086508539334317902e-01 9.22619128069999960839453478911310e-03
            4.16340541167000005584597488450527e-02 1.83396519692999998563109897986578e-01 7.74969426190300070267369392240653e-01 9.22619315730000000286281647277065e-03
            7.61163225155999989546273809537524e-01 1.94159920285199988354563060966029e-01 4.46768545588000220991631294964463e-02 9.79169917865000023760035929853984e-03
            1.94159925414399997523062779691827e-01 7.61163215393800052233075348340208e-01 4.46768591917999779994374875968788e-02 9.79170099970000026856631336613646e-03
            7.57937874717300008597931082476862e-01 4.39826608586000034062379882016103e-02 1.98079464424099987995830929321528e-01 9.88163756709999931138188600243666e-03
            4.39826512395000007726864055257465e-02 7.57937824230800027969223719992442e-01 1.98079524529700012891453297925182e-01 9.88163833384999960862682399920232e-03
            3.69760535917999996757821179471648e-02 5.36318607643599953860302775865421e-01 4.26705338764600039525021202280186e-01 9.94031955095000044342334888369805e-03
            5.36318713434200011747066128009465e-01 3.69760780935000019087333100742399e-02 4.26705208472299979405306658009067e-01 9.94032428879999992166194289211489e-03
            1.00125694892100006172519499614282e-01 7.91226709354500012949529264005832e-01 1.08647595753399994755739044194343e-01 1.03590919242000003425152598879322e-02
            7.91226669352400047152684692264302e-01 1.00125755467300001089192562631069e-01 1.08647575180299951758122745104629e-01 1.03590967446499995774367164358409e-02
            3.79866714176999992202965472642973e-02 4.15741312855800015935159308355651e-01 5.46272015726499970966756336565595e-01 1.04471535720000001379315079930166e-02
            4.15741402896500023000925239102799e-01 3.79867061534999994609762552499888e-02 5.46271890949999949782522890018299e-01 1.04471625977999998946543769307027e-02
            6.50710649146299946465887842350639e-01 4.20141226713000026649069695849903e-02 3.07275228182400050869205188064370e-01 1.07432286942500000020972450442969e-02
            4.20141133438000011213198092718812e-02 6.50710564508400035599322563939495e-01 3.07275322147799956340463722881395e-01 1.07432293003500000522221213827834e-02
            4.25548444254000013398098190009478e-02 2.92062602348399980023430089204339e-01 6.65382553226199990881184476165799e-01 1.11109066517999996448473609689245e-02
            2.92062710723999974238296317707864e-01 4.25548546752999995446131720200356e-02 6.65382434600700123361605164973298e-01 1.11109080101499998333158814034505e-02
            5.38972953818000033443524898757460e-01 4.19303146900500023352975631496520e-01 4.17238992814999432034994697460206e-02 1.11672652727500005404026595101641e-02
            4.19303182848900002355208016524557e-01 5.38972909360999952177451177703915e-01 4.17239077901000454673408057715278e-02 1.11672689369499995165613270842186e-02
            6.54947200970000031894358016870683e-01 3.00735263616200010350354432375752e-01 4.43175354137999577552875507535646e-02 1.12379462473000003158718484996825e-02
            3.00735279091700002940257263617241e-01 6.54947181273100031972944634617306e-01 4.43175396352000205979493330232799e-02 1.12379490219999995936905889948321e-02
            3.75240077158499996112794860891881e-01 3.45398013075200016608334863121854e-01 2.79361909766299931767719044728437e-01 1.14850697922500005060397754164114e-02
            3.45398028278600000007259041012730e-01 3.75240069567299994179876421185327e-01 2.79361902154100005812864537801943e-01 1.14851697218999997812671765018422e-02
            9.94532168760999951695467302670295e-02 1.59830869518699986153720260517730e-01 7.40715913605200060310096432658611e-01 1.16399188050999996546508441497281e-02
            1.59830935958500008897686939235427e-01 9.94531960132000042440481024641485e-02 7.40715868028300028491628381743794e-01 1.16399213753000004001325606850514e-02
            1.79732666166700000287548277810856e-01 7.12458543092399976259798677347135e-01 1.07808790740899995697077429213095e-01 1.34741599823499993981990030533780e-02
            7.12458446194299988007969659520313e-01 1.79732772223999998084664753150719e-01 1.07808781581700013907365587328968e-01 1.34741653553499991791131407126159e-02
            1.06606567863599996370815858881542e-01 7.00170178417499955436653635842958e-01 1.93223253718900034314742697461043e-01 1.40219379004999991633617639763543e-02
            7.00170190409600046166360698407516e-01 1.06606585567700001826452194109152e-01 1.93223224022699952007187107483333e-01 1.40219382303500006425878154914244e-02
            9.93303629801000065313587583659682e-02 6.06564798479599986968935354525456e-01 2.94104838540300006499705887108576e-01 1.43763135086000005957718173021931e-02
            6.06564805252099992216585633286741e-01 9.93303896768999994293380950693972e-02 2.94104805071000008354076271643862e-01 1.43763193635499993833226639594614e-02
            1.02322354270399998310558942193893e-01 2.53338157952800013816840873914771e-01 6.44339487776799946239236760447966e-01 1.49490414531499998807007045797945e-02
            2.53338232493799997424588354988373e-01 1.02322382618900004080941812389938e-01 6.44339384887299915227742985734949e-01 1.49490461379499992022878984698764e-02
            6.16622671521700027597034932114184e-01 2.76950206057499981326941451698076e-01 1.06427122420799991076023616187740e-01 1.54502179257999996270234177586644e-02
            2.76950069310899993357821813333430e-01 6.16622790062399950272720161592588e-01 1.06427140626700111880609256331809e-01 1.54502192977999993489524044321115e-02
            9.04184571873000025865607653940970e-02 4.98152263700100017551619657751871e-01 4.11429279112599965984031769039575e-01 1.57015508544000002844054364459225e-02
            4.98152276724800002760673578450223e-01 9.04185045148999944064627243278665e-02 4.11429218760300030588439312850824e-01 1.57015536977500001769580961763495e-02
            9.28231860167999939248417717863049e-02 3.73841851690799975571621871495154e-01 5.33334962292400072136899780161912e-01 1.59595776511999992708368267813057e-02
            3.73841869922900027489731655805372e-01 9.28232584789999981556007924154983e-02 5.33334871598100002110243167408044e-01 1.59595834189000009006864644334200e-02
            2.52167884040700007730606557743158e-01 2.52168092569700019733858198378584e-01 4.95664023389599972535535243878257e-01 1.60714962030999997111280208628159e-02
            5.08750021870799984391453563148389e-01 3.90558054433000012561905123220640e-01 1.00691923696200003046641313630971e-01 1.65197800694000009047446297927308e-02
            3.90557911673100000005121046342538e-01 5.08750143766099949260706125642173e-01 1.00691944560800106245324059273116e-01 1.65197815914500008294130140029665e-02
            1.70614146909600000734386071599147e-01 5.26673803955399977283491352864075e-01 3.02712049135000049737698191165691e-01 1.78084547794499989392669192511676e-02
            5.26673776131200033212564903806197e-01 1.70614225753699999543755438935477e-01 3.02711998115099967243679657258326e-01 1.78084638027000002891142571570526e-02
            3.48758152762900008880819768819492e-01 2.58805508488600011229863184780697e-01 3.92436338748499979889317046399810e-01 1.82870594999000012081946664466159e-02
            2.58805359601699980220956831544754e-01 3.48758349170300019359558518772246e-01 3.92436291228000055930635880940827e-01 1.82870757602000000296982307190774e-02
            1.69661455805300009913594294630457e-01 3.01352218396399973432409069573623e-01 5.28986325798300072165147867053747e-01 1.82988823495000008101385446934728e-02
            3.01352180687500026401437480672030e-01 1.69661596321900004502580827647762e-01 5.28986222990599941340406076051295e-01 1.82989026944499998117432681965511e-02
            2.58020240975899983748576005382347e-01 4.58474177447800002038746924881707e-01 2.83505581576299958701525838478119e-01 1.84972840056999987545793828758178e-02
            4.58474086019799997870904917363077e-01 2.58020381901099993271486710000318e-01 2.83505532079100008857608372636605e-01 1.84972887529500015779682087213587e-02
            1.84889868349800007285566039172409e-01 1.84889870455099997181847015781386e-01 6.30220261195099995532586945046205e-01 1.87026811893499997352385832982691e-02
            6.13074033846500054956152325758012e-01 1.92161199406899990682617840320745e-01 1.94764766746599954361229833921243e-01 1.87775129158499996806508391955504e-02
            1.92161175099400005628069720842177e-01 6.13074039838899964571794498624513e-01 1.94764785061700029800135780533310e-01 1.87775156264999992794528793638165e-02
            4.18054116059900005009097867514356e-01 1.65061333641599999877414006732579e-01 4.16884550298499911846761278866325e-01 1.94443846742999998378209625116142e-02
            1.65061264203600011057915253331885e-01 4.18054119924400002794584452203708e-01 4.16884615871999986147500294464407e-01 1.94443854170999984309897712364545e-02
            5.15920573962500017550780739838956e-01 2.98271900522899979790736324503087e-01 1.85807525514600002658482935657958e-01 1.96352821774000013688166177416861e-02
            2.98271893575000002218899908257299e-01 5.15920553436200024499669325450668e-01 1.85807552988800028792581997549860e-01 1.96352901258500002479312485093033e-02
            4.09889460234000024829725816744030e-01 4.09889431779200008598706972406944e-01 1.80221107986800022082718442106852e-01 1.99383439915499990902070237552834e-02
            ];
        
    case  25
        
        % ALG. DEG.:   25
        % PTS CARD.:  120
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.7e-13
        
        xyw_bar=[
            8.28815950330000043855260827285747e-03 9.84820276886899992341284360009013e-01 6.89156360980003324101517137023620e-03 7.43670892949999952879769349323169e-04
            4.61842203024100006469865320468671e-01 5.38157796975900049041285910789156e-01 -1.11022302462515654042363166809082e-16 7.44451763099999996864397644458222e-04
            7.10664412390000013730606553963298e-03 8.08423613900000059828698795172386e-03 9.84809119737099947222702667204430e-01 7.50297219000000022159913015684651e-04
            9.84761314169899981685318834934151e-01 7.00157551339999992962015440411960e-03 8.23711031670001925242274865013314e-03 7.52960415649999987956764169183543e-04
            5.37444786904900051460742815834237e-01 4.62555213095100004050408415423590e-01 -5.55111512312578270211815834045410e-17 7.65943435750000042698537594532127e-04
            0.00000000000000000000000000000000e+00 4.88767688014000001306413878410240e-01 5.11232311986000054204737352847587e-01 1.15163172435000008518579672767146e-03
            4.91413192936099973984198641119292e-01 0.00000000000000000000000000000000e+00 5.08586807063900026015801358880708e-01 1.18245335209999995716712017923555e-03
            7.03459370200000023987119135426838e-03 9.57415805369699945970296539599076e-01 3.55496009283000624634496489306912e-02 1.43755718054999997827791968774136e-03
            9.56473471422800036734201967192348e-01 3.64655449485000032749937304288324e-02 7.06098362869995999080430237881956e-03 1.49312443675000006831909971083405e-03
            3.70198792044999988593012574256136e-02 7.09085771659999971466836754530050e-03 9.55889263078899986680880829226226e-01 1.51920813685000005537062417459993e-03
            1.02412454274699998846287485321227e-01 8.93612559493699998292015607148642e-01 3.97498623159997510612129190121777e-03 1.60462298440000009254469137687238e-03
            5.92806581150900013810201016895007e-01 4.94517055999999996518390332767012e-03 4.02248248289099996632955935638165e-01 1.85147992174999995347584036409216e-03
            5.09484223709999982376839255948653e-03 9.96676659189000002658787025211495e-02 8.95237491844000032870098948478699e-01 1.87035930175000002749496896825576e-03
            8.15620236890000037399683918692972e-03 4.15561148783999967371016737160971e-02 9.50287682752700035848647530656308e-01 1.92262716114999991472156448679698e-03
            4.24936107567999987733031730385846e-02 9.49486526035200006923275850567734e-01 8.01986320799996654784536076476797e-03 1.93353893339999992340150214431560e-03
            9.49554350084400033971121501963353e-01 8.17945072920000029736975477590022e-03 4.22661991863999692009556952143612e-02 1.95962775889999994347046019527170e-03
            8.93278747123899985460582229279680e-01 5.32243262620000004425868667112809e-03 1.01398820249900017964606036002806e-01 1.97866413440000001380814076412662e-03
            6.93176129270000034726662008210951e-03 9.06540102043300044520890423882520e-01 8.65281366639999438561403621861245e-02 2.20161258619999992622306983491853e-03
            9.03583903066500049305886932415888e-01 8.94771171076999966009779541309399e-02 6.93897982579995409313511345317238e-03 2.29535540864999991483852426199519e-03
            9.05665738208999981928570832678815e-02 7.05253420050000034285941907796769e-03 9.02380891978599941616323576454306e-01 2.35118347174999992033050943973649e-03
            8.39293327870000056833976742609593e-03 6.66317993111099959868681708030635e-01 3.25289073610199985786550769262249e-01 2.50074219090000004345197837096748e-03
            6.26124568607099973860385944135487e-01 9.21975831529999942381969901816774e-03 3.64655673077600051001923020521645e-01 2.61939150779999995952151969902388e-03
            6.28015929790000024257601296540088e-03 8.33520746052699990080725456209620e-01 1.60199094649399964573888155427994e-01 2.72110520459999984110854853724959e-03
            8.27253925736699979864852139144205e-01 1.66513493932999995239541135561012e-01 6.23258033030002489560672529478325e-03 2.84656244560000000476396309068150e-03
            6.20058753530000009851486098000350e-03 7.42469325522900014391325385076925e-01 2.51330086941800034949778819282074e-01 2.95537114945000015803211468323752e-03
            1.67690031118500004758686827699421e-01 6.57177435279999999639954921804019e-03 8.25738194528700009122701430896996e-01 2.98439838434999982902384729754885e-03
            7.19935306956700049596520329941995e-01 6.43545349620000019241761179955574e-03 2.73629239547099945006891630328028e-01 3.36310951435000006840003727859312e-03
            2.74974009023700005638346510750125e-01 7.18529612071900025682680279714987e-01 6.49637890439991316782197827706113e-03 3.41539243119999991887825174785576e-03
            7.92575820050000004468593317596969e-03 1.76641137471400005987121062389633e-01 8.15433104328100011315427764202468e-01 3.47656295560000005301359848886023e-03
            6.99812207519999972260560383574557e-03 2.70476725400400019871938184223836e-01 7.22525152524399971731838832056383e-01 3.62301353209999992682766034590713e-03
            8.12524877326299965751843501493568e-01 8.22995332100000080710611172207791e-03 1.79245169352700045584114718622004e-01 3.63640948065000000483104614090735e-03
            7.35369699699999976316711070012389e-03 5.93416787545300028483552523539402e-01 3.99229515457700023794984645064687e-01 3.65044654234999987885146310873097e-03
            7.28366593541100004749466734210728e-01 2.64881755375199978708167236618465e-01 6.75165108370001654236602917080745e-03 3.68023333879999997586551785389020e-03
            1.80064230456499996257946349942358e-01 8.11584897668199967135649330884917e-01 8.35087187530003660640431917272508e-03 3.70599616275000013840279500243469e-03
            2.65810246776200020946134827681817e-01 6.85535254289999958221546449976813e-03 7.27334400680899939573009760351852e-01 3.74461071680000021077994176721404e-03
            7.08923645200000042410071188214715e-03 3.75763265974400018087919761455851e-01 6.17147497573599990161596906546038e-01 3.93020336300000001916732372819752e-03
            3.77405430204300007535067607022938e-01 6.14857353375700022013461421011016e-01 7.73721642000002596262220322387293e-03 3.93108632115000012491323744256988e-03
            3.69649608667999979294549461883435e-02 9.21079230289300054934642503212672e-01 4.19558088438999332581147427845281e-02 4.02531805329999978204646993162896e-03
            9.20319410980499963415013553458266e-01 4.26025082113999989386066147289966e-02 3.70780808081000376463798318127374e-02 4.07214302364999978373072408999178e-03
            4.25477806430999971887274568871362e-02 3.72689941794000015318211183057429e-02 9.20183225177499952707194097456522e-01 4.07394020760000041131432269025936e-03
            6.19127839498299969989147939486429e-01 3.72405571380900024536231285310350e-01 8.46658912080000547462077520322055e-03 4.62220733060000020520075025842743e-03
            3.76269720917799976867001987557160e-01 8.14364220110000078256540234633576e-03 6.15586636881100046636561273771804e-01 4.73373175824999981725627051787342e-03
            9.56111149690000022838276549919101e-02 8.77109837260099944877822508715326e-01 2.72790477709000667161376441072207e-02 4.85661050685000025839421766704618e-03
            3.02473410376999998772085120890551e-02 9.43858903393000048254180001094937e-02 8.75366768623000002236267391708679e-01 4.98767905755000042627989031984725e-03
            8.73990569175399967249973087746184e-01 3.13198990882999975249312285541237e-02 9.46895317363000421639895876069204e-02 5.16839018364999985832852757994260e-03
            8.60413373495800049006732024281519e-01 1.04901978204599996558066266061360e-01 3.46846482995999544352017096571217e-02 5.61316385829999965617620460989201e-03
            3.47307852351999984952257705117518e-02 8.60985646288599970610277978266822e-01 1.04283568476200017016708443406969e-01 5.71545593725000019430515507679047e-03
            1.04360660834299998067642434307345e-01 3.57152881003999986275765365917323e-02 8.59924051065300010243674933008151e-01 5.77752837435000030780196667024029e-03
            7.79762282475400025916201229847502e-01 1.87231819926500009376724165122141e-01 3.30058975980999647070746050303569e-02 6.77879284784999990715048667766496e-03
            1.85865164256000005293945065432126e-02 4.83439767879400017491775543021504e-01 4.97973715694999929937125671131071e-01 6.79924814499999977807087603309810e-03
            3.24585286617999971281456339511351e-02 7.78347491604200025783200089790625e-01 1.89193979733999984027548180165468e-01 6.88774069184999986753537015715665e-03
            8.37129390115699978380803258914966e-01 8.04060570155999965180626531946473e-02 8.24645528687000251011340878903866e-02 6.89805079709999990900115562908468e-03
            8.36602075314999971134000134043163e-02 8.42141481705099947951964622916421e-01 7.41983107633999994234841324214358e-02 6.92044199519999993219787626230755e-03
            7.84070242501000019830215137517371e-02 8.49927089145000036474186799750896e-02 8.36600266835400008247347614087630e-01 7.03170099885000021017189553163007e-03
            4.92923864845799974965245837665861e-01 4.89285591470999980945322249681340e-01 1.77905436832000440894319126527989e-02 7.04957255045000029392010532092172e-03
            1.87063758407299995933925629287842e-01 3.45210858280999982605763420906442e-02 7.78415155764599964172134605178144e-01 7.10020559954999998242275438542492e-03
            4.89263696702499972790434412672766e-01 1.90774755077000000846876304194666e-02 4.91658827789800079166582236211980e-01 7.22592122584999989320930779967966e-03
            4.01982618372000027884283213097660e-02 1.69114318710899991238250095193507e-01 7.90687419451900019851109391311184e-01 7.51229898195000019706091265447867e-03
            7.89425927886500011787518360506510e-01 4.12206731483999999632317212672206e-02 1.69353398965099988249249918226269e-01 7.64089020609999992200700447142481e-03
            1.68626045642900013188381080908584e-01 7.89486064058499970030879921978340e-01 4.18878902985999612695877658552490e-02 7.77753620845000014083492345662307e-03
            3.75090191317400012405158804540406e-01 5.89531827201300040464104768034304e-01 3.53779814812998916195851961674634e-02 8.22854429999999963107892853031444e-03
            3.56362876879999998513959269530460e-02 3.68125621769899991075902789816610e-01 5.96238090542100085400534226209857e-01 8.26378797864999944944308651884057e-03
            5.88754816480399956191149613005109e-01 3.59968962540999984356382412897801e-02 3.75248287265500024556530433983426e-01 8.34237772255000051280227069128159e-03
            3.73308082181999997817278824641107e-02 6.79070467353300011303929295536363e-01 2.83598724428499981975448918092297e-01 8.37046564924999979273589900685693e-03
            2.82076999337399991762964646113687e-01 3.73639992361000025655748402186873e-02 6.80559001426500054243717841018224e-01 8.43373316804999981211032888950285e-03
            6.81927760331999976983752276282758e-01 2.80333034572499972814085822392371e-01 3.77392050955000502021619013248710e-02 8.44411150825000005049059126349675e-03
            3.74938324382000018331062562992884e-02 2.63401618001399995261380126976292e-01 6.99104549560400023722195328446105e-01 8.60435563455000038124431682717841e-03
            6.98407920412699945167389614653075e-01 3.64154673321999966528750292127370e-02 2.65176612255100030424159740505274e-01 8.73405341320000055960193208193232e-03
            2.65439089407900008499296973241144e-01 6.98071743619300022487550450023264e-01 3.64891669728000245243038079934195e-02 8.83319498069999944900310140383226e-03
            1.42984844080000006449893135140883e-01 7.61225461845300044316786625131499e-01 9.57896940746999492333202397276182e-02 9.14838107374999973120388574443496e-03
            7.62355400764699986382311180932447e-01 9.43741220274999931261206143062736e-02 1.43270477207800006613780396946822e-01 9.17884262295000016851354018854181e-03
            9.34222022749000013952525023341877e-02 1.47979983683200005950553190814389e-01 7.58597814041899964898618691222509e-01 9.31962847604999970740902170973641e-03
            5.75900447992300001942567178048193e-01 3.82132964169800026787982005771482e-01 4.19665878378999712694508161803242e-02 9.48905302950000043016309803078912e-03
            3.82242733252500022089748199505266e-01 4.26716362301000004531026377208036e-02 5.75085630517400026029406490124529e-01 9.59239612889999961053977273195414e-03
            4.11414081674999973503581429667975e-02 5.71808287443200002897469857998658e-01 3.87050304389299992813278095127316e-01 9.70402210219999984686101157649318e-03
            8.02462538379000028809784339500766e-02 7.70220438204200053000647585577099e-01 1.49533307957899985751737403916195e-01 9.73600360965000063973029398312065e-03
            7.62522981940999988736962222901639e-01 1.55942057736200012652005852942239e-01 8.15349603227999986110319241561228e-02 1.00427540247499998832259038294978e-02
            1.52494144513100005067229858468636e-01 8.42965421322000030279752991191344e-02 7.63209313354699991904794842412230e-01 1.00836954666000001151759946083075e-02
            6.22159195833000000508405946675339e-02 4.53818131887300024729370306886267e-01 4.83965948529400002975364714075113e-01 1.10871081380500002833144890246331e-02
            1.10953903607600004588285003137571e-01 4.58601407117100012822419330404955e-01 4.30444689275299940955932243014104e-01 1.14851220254000001980143608193430e-02
            4.57562721205700018156647956857341e-01 4.79531356020999999234533106573508e-01 6.29059227732999270976677053113235e-02 1.16732558699499996274218105440923e-02
            4.32286513637400005016786508349469e-01 1.23059123747200002685353581455274e-01 4.44654362615399978420072102380800e-01 1.17441567669000004880963672349026e-02
            5.86500285024100009856340420810739e-01 8.34119779793000004630698640539777e-02 3.30087736996599989680589715135284e-01 1.20341049508999996947578026151859e-02
            8.69359250817999945626368685225316e-02 6.75567701335100023740665164950769e-01 2.37496373583099940063334543083329e-01 1.20455396476500005420451344662069e-02
            9.29594906936000026087540959451871e-02 2.32650089272700005826877145409526e-01 6.74390420033699977686580950830830e-01 1.22838524740500002613297070297449e-02
            6.66193214145400003367569752299460e-01 2.44829400740600000574431760469452e-01 8.89773851139999960579984872310888e-02 1.23268157859499991740870328271740e-02
            4.78030636222699978343797511115554e-01 6.61749044834999966546718042081920e-02 4.55794459293799969490379453418427e-01 1.23378265025999994675132143129304e-02
            4.37221529457699986664920288603753e-01 4.44214558524400005001808722226997e-01 1.18563912017900008333270989169250e-01 1.24852301355000000482320388073276e-02
            6.77922450466900028409611422830494e-01 9.29096534576999977250721940436051e-02 2.29167896075399973865316383125901e-01 1.25013272041000003675925711377204e-02
            2.42343125565999989445842288660060e-01 8.89793655129000049530674232300953e-02 6.68677508921100005601090288109845e-01 1.25245434712999998555504888031464e-02
            2.28892542030499995808767721428012e-01 6.78005308167199949309633666416630e-01 9.31021498023000271260229965264443e-02 1.25468125062499994554210402952776e-02
            3.31506504995900008747611309445347e-01 5.84738155974099949041544732608600e-01 8.37553390299999866996927266882267e-02 1.25741038113000008075159641407481e-02
            3.42420052660700025004558710861602e-01 5.13924572273599999938653581921244e-01 1.43655375065700030567938938474981e-01 1.27505145223499993106797134601038e-02
            8.62630046475000022976331592872157e-02 3.34097624923400005059193063061684e-01 5.79639370429099964887598162022186e-01 1.28272255989500000988856598382881e-02
            5.11318894663500000952183199842693e-01 1.38015472055399990303214963205392e-01 3.50665633281100008744601836951915e-01 1.28987375315000006581467317801071e-02
            1.53897784100099993542798415546713e-01 6.78806261956199974427761389961233e-01 1.67295953943700004273864578863140e-01 1.35003876996500002394263617588877e-02
            6.77995134847199976135811994026881e-01 1.66335892526900008814294551484636e-01 1.55668972625900015049893454488483e-01 1.37215768421999999293614536099994e-02
            1.66460046941100003881075508616050e-01 1.58221450484899994748388962761965e-01 6.75318502574000056881686759879813e-01 1.38536200744000005669054331747247e-02
            9.50910318887999983505920909010456e-02 5.66659033254300004145420643908437e-01 3.38249934856900025259562880819431e-01 1.39142207681999998530875117808137e-02
            3.43604813671199993407867623318452e-01 9.78960873457000030350982910931634e-02 5.58499098983100017434821893402841e-01 1.43603690552500003196900735247254e-02
            5.56041702536600013218048843555152e-01 3.46891782094700007466059332728037e-01 9.70665153686999793158918237168109e-02 1.44413417478000006416438694145654e-02
            1.45240402951300001666012917667103e-01 3.59953449105199996260751049703686e-01 4.94806147943499974317660417000297e-01 1.46651364879500003407519770348699e-02
            1.61968515623799991232090178527869e-01 5.81013137332999973772018620365998e-01 2.57018347043200034995891201106133e-01 1.59451439778500007882211519927296e-02
            5.80016484426200018553743120719446e-01 2.56067464067199990207512882989249e-01 1.63916051506599991238743996291305e-01 1.59541830142999989383678638432684e-02
            2.45020122328799994093628811242525e-01 5.88146955210199995356390445522266e-01 1.66832922461000010549980743235210e-01 1.60469480164499994634397239678947e-02
            2.55762189179400023508748063250096e-01 1.65224406504699994346907487852150e-01 5.79013404315900093166646911413409e-01 1.60809304389999997708660117723412e-02
            2.20523998551099997822433351757354e-01 3.49650746610600027253212829236872e-01 4.29825254838299919413202587747946e-01 1.61212063767000000902296363847199e-02
            4.94018311128499976181416286635795e-01 2.54944844845300000457655187346973e-01 2.51036844026200078872079757275060e-01 1.63536223210500007707590697236810e-02
            2.53157068979800004004943048130372e-01 2.54336911501700013715776549361181e-01 4.92506019518500037790431633766275e-01 1.64973158347500002685048769990317e-02
            5.84689111635700031754936389916111e-01 1.66660391647900002487503456904960e-01 2.48650496716399965757560153178929e-01 1.65914048012500009443570547773561e-02
            1.66033360227799992037134302336199e-01 2.52324019170500013498070757123060e-01 5.81642620601700022220370556169655e-01 1.67428581325500015763996231044075e-02
            2.50542629246099990414364810931147e-01 4.95900762752799983967832986309077e-01 2.53556608001100081128953434017603e-01 1.67734236396000005453466030758136e-02
            3.51933680218199984590654594285297e-01 1.80538036779999994108081295962620e-01 4.67528283001799938034537262865342e-01 1.68524521493999997201473917129988e-02
            3.50266883541899998188995368764154e-01 4.35858232988100002813780520227738e-01 2.13874883469999943486072879750282e-01 1.70180731383499997400043923789781e-02
            4.40089248551200018511053713154979e-01 2.12057610494099990638972030865261e-01 3.47853140954699990849974255979760e-01 1.71232617661499983419126635908469e-02
            4.68085547154599990182788360471022e-01 3.55268157077400004162370805715909e-01 1.76646295768000061165992065070895e-01 1.72764408625500007210540331925586e-02
            1.77023776394699988712133631452161e-01 4.67035292226600007303716211026767e-01 3.55940931378699976228574541892158e-01 1.78391437851499988498371607192894e-02
            3.90092077950099980210296735094744e-01 3.32315281929999983834278509675642e-01 2.77592640119899980444273523971788e-01 1.82328112508000003688657386646810e-02
            2.80584777411999997109859350530314e-01 3.89804117667999994178984479731298e-01 3.29611104919999953200004938480561e-01 1.82586354352999996608630794980854e-02
            3.36152334744000025334997872050735e-01 2.77850004435599984287108554781298e-01 3.85997660820399934866742341910140e-01 1.85962405508999992220786623420281e-02
            ];
        
        
    otherwise
        xyw=[];
end




pointset_stats=[
    2    3 1.73e-13    0    0
    4    6 1.87e-13    0    0
    5   10 1.77e-13    0    0
    7   15 4.59e-13    0    0
    9   21 3.33e-13    0    0
    11   28 5.89e-13    0    0
    13   36 3.94e-13    0    0
    14   45 4.53e-13    0    0
    16   55 7.82e-13    0    0
    18   66 5.01e-13    0    0
    20   78 7.68e-13    0    0
    21   91 8.94e-13    0    0
    23  105 7.02e-13    0    0
    25  120 5.66e-13    0    0
    ];





