
function [xyw,pointset_stats]=set_taylor_wingate_bos_standard(deg)

%--------------------------------------------------------------------
% CUBATURE RULE ON THE SIMPLEX [0 0; 0 1; 1 0] WITH ADE EQUAL TO deg.
% THE CUBATURE RULE IS DESCRIBED IN STANDARD FORM.
%--------------------------------------------------------------------
% INPUT:
% deg: algebraic degree of precision (ADE) of the rule.
%--------------------------------------------------------------------
% OUTPUT:
% xyw: MATRIX OF NODES xyw(:,1:2) AND WEIGHTS xyw(:,3).
%      REMARK: The sum of the weights makes 1/2.
%
% pointset_stats:
%    statistics of the pointset stored in a matrix. Each rows contains
%    degree, cardinality of the full pointset, moment error w.r.t. orthn.
%    Dubiner basis,number of points with negative weights, number of
%    points outside the domain.
%--------------------------------------------------------------------
% SOURCE:
%  Mark A. Taylor, Beth A. Wingate, Len P. Bos,
%  "Several new quadrature formulas for polynomial integration in the triangle",
%  unpublished note, see arXiv.
%--------------------------------------------------------------------
% | DEG | CARD | W- | PT.O|   M.E.   |Q.|
%--------------------------------------------------------------------
% |   2 |   3 |   0 |   0 | 1.73e-13 |PI|
% |   4 |   6 |   0 |   0 | 1.87e-13 |PI|
% |   5 |  10 |   0 |   0 | 1.77e-13 |PI|
% |   7 |  15 |   0 |   0 | 4.59e-13 |PI|
% |   9 |  21 |   0 |   0 | 3.33e-13 |PI|
% |  11 |  28 |   0 |   0 | 5.89e-13 |PI|
% |  13 |  36 |   0 |   0 | 3.94e-13 |PI|
% |  14 |  45 |   0 |   0 | 4.53e-13 |PI|
% |  16 |  55 |   0 |   0 | 7.82e-13 |PI|
% |  18 |  66 |   0 |   0 | 5.01e-13 |PI|
% |  20 |  78 |   0 |   0 | 7.68e-13 |PI|
% |  21 |  91 |   0 |   0 | 8.94e-13 |PI|
% |  23 | 105 |   0 |   0 | 7.02e-13 |PI|
% |  25 | 120 |   0 |   0 | 5.66e-13 |PI|
%--------------------------------------------------------------------
% LEGEND:
%--------------------------------------------------------------------
% DEG  : algebraic degree of precision
% CARD : cardinality of the pointset
% W-   : negative weights
% PT.O : points outside the simplex [0 0; 1 0; 0 1].
% M.E. : maximum moment error in inf norm.
% Q.: quality of the rule.
%        PI: pos. weights, points in simplex
%        NI: some neg. weights, points in simplex
%        P0: pos. weights, some points out of the simplex
%        N0: some neg. weights, some points out of the simplex
%--------------------------------------------------------------------


switch deg
    
    case   2
        
        % ALG. DEG.:    2
        % PTS CARD.:    3
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.7e-13
        
        xyw=[
            1.66666666666699991861122498448822e-01 6.66666666666700047372273729706649e-01 1.66666666666675011843068432426662e-01
            6.66666666666700047372273729706649e-01 1.66666666666699991861122498448822e-01 1.66666666666675011843068432426662e-01
            1.66666666666699991861122498448822e-01 1.66666666666699991861122498448822e-01 1.66666666666675011843068432426662e-01
            ];
        
    case   4
        
        % ALG. DEG.:    4
        % PTS CARD.:    6
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.9e-13
        
        xyw=[
            9.15762135097999974142624068917939e-02 9.15762135097999974142624068917939e-02 5.49758718276499996902018096989195e-02
            8.16847572980500036265993912820704e-01 9.15762135097999974142624068917939e-02 5.49758718276499996902018096989195e-02
            9.15762135097999974142624068917939e-02 8.16847572980500036265993912820704e-01 5.49758718276499996902018096989195e-02
            1.08103018168100001306086710428644e-01 4.45948490915999973260852584644454e-01 1.11690794839000004379236941076670e-01
            4.45948490915999973260852584644454e-01 1.08103018168100001306086710428644e-01 1.11690794839000004379236941076670e-01
            4.45948490915999973260852584644454e-01 4.45948490915999973260852584644454e-01 1.11690794839000004379236941076670e-01
            ];
        
    case   5
        
        % ALG. DEG.:    5
        % PTS CARD.:   10
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.8e-13
        
        xyw=[
            0.00000000000000000000000000000000e+00 1.00000000000000000000000000000000e+00 6.56780248759999962793187933129957e-03
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 6.56791530169999974303474132852898e-03
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 6.85409869000000015332174641002894e-03
            2.67327353118499977391309130325681e-01 6.72819921871000037327803511288948e-01 5.87095966455749990187129583318892e-02
            6.72817552946100017052799557859544e-01 2.67328859948199992757622567296494e-01 5.87103059567000012952675547239778e-02
            6.49236350053999933251702714187559e-02 6.71653011149400014367927269631764e-01 6.20062948278500003507218707454740e-02
            6.71649853904199956033949092670809e-01 6.49251690029000044779294853469764e-02 6.20076230630249997100023051643802e-02
            6.54032456799999994956351656583138e-02 2.69378936645299993379865099996096e-01 6.29651151382250029442388949973974e-02
            2.69376706914000008108445172183565e-01 6.54054874919000051036732656939421e-02 6.29665133414499950115228443792148e-02
            3.38673850389600017418700872440240e-01 3.38679989302699990805223251300049e-01 1.12644734547849997841062474890350e-01
            ];
        
    case   7
        
        % ALG. DEG.:    7
        % PTS CARD.:   15
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.6e-13
        
        xyw=[
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 2.56395435230000016957330188915876e-03
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 2.56395435230000016957330188915876e-03
            0.00000000000000000000000000000000e+00 1.00000000000000000000000000000000e+00 2.56395435230000016957330188915876e-03
            7.83965665101200026576577783998800e-01 4.21382841641999988935296528325125e-02 2.79011761661750005980575650710307e-02
            1.73896050734500012824312875636679e-01 7.83965665101200026576577783998800e-01 2.79011761661750005980575650710307e-02
            1.73896050734500012824312875636679e-01 4.21382841641999988935296528325125e-02 2.79011761661750005980575650710307e-02
            4.21382841641999988935296528325125e-02 1.73896050734500012824312875636679e-01 2.79011761661750005980575650710307e-02
            7.83965665101200026576577783998800e-01 1.73896050734500012824312875636679e-01 2.79011761661750005980575650710307e-02
            4.21382841641999988935296528325125e-02 7.83965665101200026576577783998800e-01 2.79011761661750005980575650710307e-02
            4.74388086175200007765795362502104e-01 4.74388086175200007765795362502104e-01 4.19943898833749981736040979285463e-02
            4.74388086175200007765795362502104e-01 5.12238276497000016851401937856281e-02 4.19943898833749981736040979285463e-02
            5.12238276497000016851401937856281e-02 4.74388086175200007765795362502104e-01 4.19943898833749981736040979285463e-02
            2.38561530018099987593416244635591e-01 5.22876939963899944885383774817456e-01 6.63059700986499966912646186756319e-02
            5.22876939963899944885383774817456e-01 2.38561530018099987593416244635591e-01 6.63059700986499966912646186756319e-02
            2.38561530018099987593416244635591e-01 2.38561530018099987593416244635591e-01 6.63059700986499966912646186756319e-02
            ];
        
    case   9
        
        % ALG. DEG.:    9
        % PTS CARD.:   21
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.3e-13
        
        xyw=[
            4.51890097843999985482987824525480e-02 4.51890097843999985482987824525480e-02 1.29967855161499997590501820354802e-02
            4.51890097843999985482987824525480e-02 9.09621980431200016781190242909361e-01 1.29967855161499997590501820354802e-02
            9.09621980431200016781190242909361e-01 4.51890097843999985482987824525480e-02 1.29967855161499997590501820354802e-02
            7.47512472733899979715488370857202e-01 3.04243617287999999765357728165327e-02 1.76758525445999993552792517448324e-02
            2.22063165537299989082953288743738e-01 3.04243617287999999765357728165327e-02 1.76758525445999993552792517448324e-02
            7.47512472733899979715488370857202e-01 2.22063165537299989082953288743738e-01 1.76758525445999993552792517448324e-02
            2.22063165537299989082953288743738e-01 7.47512472733899979715488370857202e-01 1.76758525445999993552792517448324e-02
            3.04243617287999999765357728165327e-02 7.47512472733899979715488370857202e-01 1.76758525445999993552792517448324e-02
            3.04243617287999999765357728165327e-02 2.22063165537299989082953288743738e-01 1.76758525445999993552792517448324e-02
            1.36991201264899997713797574760974e-01 2.18290070971399990940042812326283e-01 2.27347690237999985884975728822610e-02
            6.44718727763699983590583997283829e-01 2.18290070971399990940042812326283e-01 2.27347690237999985884975728822610e-02
            1.36991201264899997713797574760974e-01 6.44718727763699983590583997283829e-01 2.27347690237999985884975728822610e-02
            2.18290070971399990940042812326283e-01 6.44718727763699983590583997283829e-01 2.27347690237999985884975728822610e-02
            2.18290070971399990940042812326283e-01 1.36991201264899997713797574760974e-01 2.27347690237999985884975728822610e-02
            6.44718727763699983590583997283829e-01 1.36991201264899997713797574760974e-01 2.27347690237999985884975728822610e-02
            3.69603304334000029585105551177548e-02 4.81519834783300015867979482209194e-01 2.58086012845000004056750242398266e-02
            4.81519834783300015867979482209194e-01 3.69603304334000029585105551177548e-02 2.58086012845000004056750242398266e-02
            4.81519834783300015867979482209194e-01 4.81519834783300015867979482209194e-01 2.58086012845000004056750242398266e-02
            4.03603979817900027082799851996242e-01 1.92792040364099998006608416289964e-01 4.70400367291749985088067376182153e-02
            4.03603979817900027082799851996242e-01 4.03603979817900027082799851996242e-01 4.70400367291749985088067376182153e-02
            1.92792040364099998006608416289964e-01 4.03603979817900027082799851996242e-01 4.70400367291749985088067376182153e-02
            ];
        
    case  11
        
        % ALG. DEG.:   11
        % PTS CARD.:   28
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.9e-13
        
        xyw=[
            0.00000000000000000000000000000000e+00 9.45170445017399996068263590132119e-01 2.85206235082499995336924314415228e-03
            9.45170445017299964973744863527827e-01 0.00000000000000000000000000000000e+00 2.85206235082499995336924314415228e-03
            9.28900240571900015495998559345026e-01 6.85505797224000013034839184911107e-02 3.31728214299999981870548104723184e-03
            6.85505797224000013034839184911107e-02 9.28900240571699953306961106136441e-01 3.31728214299999981870548104723184e-03
            2.43268355615000007874382248473921e-02 2.43268355616000006569343838691566e-02 3.89664433375000018172640281477470e-03
            1.27966283533500008173788842213980e-01 2.77838749488000004428389644317576e-02 1.02068695107000000088515179186288e-02
            2.77838749488000004428389644317576e-02 1.27966283533699987096099448535824e-01 1.02068695107250008435872956624735e-02
            2.87083428359999996704043923045901e-02 7.49834758865699946284166799159721e-01 1.44962416278999993435316895329379e-02
            7.49834758865600026211950535071082e-01 2.87083428359999996704043923045901e-02 1.44962416278999993435316895329379e-02
            7.22800790970700046678132366650971e-01 2.49760206238500009900960208142351e-01 1.50346311915750003446179405841576e-02
            2.49760206238600013239903319117730e-01 7.22800790970700046678132366650971e-01 1.50346311915750003446179405841576e-02
            8.65562992839000039380081830131530e-02 8.32551385699699997289258135424461e-01 1.56318472108250006158414180390537e-02
            8.32551385699800028383776862028753e-01 8.65562992839000039380081830131530e-02 1.56318472108250006158414180390537e-02
            3.06161915767199976912138481566217e-01 3.03526617491000003101753179635125e-02 1.59921080376000009082293473738901e-02
            3.03526617491000003101753179635125e-02 3.06161915767500014684543430121266e-01 1.59921080376000009082293473738901e-02
            4.86861059504699988931264442726388e-01 4.86861059504699988931264442726388e-01 1.65331468040250013917802363039300e-02
            6.65790429301699959729887723369757e-01 1.76545615421900004005095752290799e-01 1.67125809205000003254681928410719e-02
            1.76545615422100010682981974241557e-01 6.65790429301600039657671459281119e-01 1.67125809205250011602039705849165e-02
            2.93121007360000011832124044985903e-02 5.29565748866899999036661483842181e-01 1.71726076494249993431573386715172e-02
            5.29565748866700047869926493149251e-01 2.93121007360000011832124044985903e-02 1.71726076494249993431573386715172e-02
            1.44467382439100011648491772575653e-01 1.44467382439100011648491772575653e-01 2.50679385964749992798950017913739e-02
            3.29974011141099976551771533195279e-01 5.36181572905000014550580544892000e-01 2.85784196024750011821957684787776e-02
            5.36181572905199965717315535584930e-01 3.29974011140900025385036542502348e-01 2.85784196024750011821957684787776e-02
            5.51150751686199980028391109954100e-01 1.43779086192300009638955771151814e-01 3.05912036687999988604502732414403e-02
            1.43779086192300009638955771151814e-01 5.51150751686199980028391109954100e-01 3.05912036687999988604502732414403e-02
            3.34806658732699990999037709116237e-01 1.52961943716100001733337876430596e-01 3.48605583544499969539387507211359e-02
            1.52961943716100001733337876430596e-01 3.34806658732699990999037709116237e-01 3.48605583544499969539387507211359e-02
            3.43018349814699974320575392994215e-01 3.43018349814699974320575392994215e-01 4.36094457295500004256894044374349e-02
            ];
        
    case  13
        
        % ALG. DEG.:   13
        % PTS CARD.:   36
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.9e-13
        
        xyw=[
            2.42935351589999987131918146587850e-02 9.49305929384600011289307985862251e-01 4.15602496892499961161915322804816e-03
            2.65193427721999996915513975181966e-02 2.42695130639999992472333190107747e-02 4.17029249444999996349059401268278e-03
            9.49212602355099965834028807876166e-01 2.65067966436999986890032943165352e-02 4.17076422667500036622900694283089e-03
            3.37757637490000012236990301062178e-03 4.76731641236300018871219208449475e-01 4.39202175207499979220360231124687e-03
            4.75767229810100000797490338300122e-01 5.19892182910200006951129125809530e-01 4.61186654612500031291677515810079e-03
            5.19078319347100025638042097853031e-01 5.59127062019999961084426587376583e-03 4.94856025470000009114368566542908e-03
            8.61683974532099994370071271987399e-01 1.33996048618000001412253752164361e-02 5.08850989637500000783987275099207e-03
            1.24920975992600005732846568662353e-01 8.61305432133400050709326478681760e-01 5.17132159849999998441072435184651e-03
            1.38565453861000005553183456186161e-02 1.24773371735799998294424995037843e-01 5.20678415214999995574984836821386e-03
            2.11887064221999987800870712817414e-02 8.43843835122299990025851457176032e-01 7.94549445697500036911442578002607e-03
            8.43229678721900022253521456150338e-01 1.35456364583000005152868538971234e-01 8.01180088102500069513478564431352e-03
            1.35423179786500008647820436635811e-01 2.13482820656000008008312818219565e-02 8.01519202864999988356053961524594e-03
            3.08885351067899993626753030184773e-01 2.21919663013999991785230037066867e-02 1.07691489795750001967622466736429e-02
            6.68505759516900055494659227406373e-01 3.08901287938899982155049883658648e-01 1.09618353834749996528197257816828e-02
            2.26545012556999998676321439461390e-02 6.69170994332100033119559157057665e-01 1.09802418183249997363226668767311e-02
            2.80851540877199989942880620219512e-01 6.92471815510599997267604521766771e-01 1.19987980922749998963228179604812e-02
            6.92244674905099999229207696771482e-01 2.68723345026000010049571642412047e-02 1.20951565183250002444292903192036e-02
            2.68617447119000006539035751984557e-02 2.81009397322199983015877933212323e-01 1.21216855843749998777880705347343e-02
            1.14177848546999999301831962839060e-01 7.97358141358599947778884597937576e-01 1.39241122006000007937931428614320e-02
            7.97480792206100019825498748105019e-01 8.79806508790999947100530675925256e-02 1.40256591088999994776376212257674e-02
            8.92807293894000014722323044225050e-02 1.14502056112799996401285795855074e-01 1.41297530923250001799740260821636e-02
            1.05248789245500004585487374697550e-01 6.68690411992200028912236575706629e-01 1.72322472667499997089901597746575e-02
            6.66302228073999969204521676147124e-01 2.27505163183199993204297584270535e-01 1.79303334022249995227671348629883e-02
            2.30780373754700007804530059729586e-01 1.05457256122099995998553367826389e-01 1.81863480244000008667626389069483e-02
            1.70505915754000009210145094584732e-01 5.17406439865800038546694850083441e-01 1.97201834184249982784287169579329e-02
            5.08659397304299987219167178409407e-01 3.17052385520899981941056466894224e-01 2.02528586377999988976661427386716e-02
            3.14182386228100019032183354283916e-01 1.81070636165899995217998252883262e-01 2.06431324763750000117479288519462e-02
            4.61746081786399975577239729318535e-01 4.67859453980400019634799946288695e-01 2.10511141832500012216478069149161e-02
            6.93087496081000037362329635470815e-02 4.62285604208500011669258356050705e-01 2.10896383326250012879565787216052e-02
            4.65195525926800024230090002674842e-01 7.24357805669000043291561041769455e-02 2.12992467122000012080818009962968e-02
            2.57862585789299991745338047621772e-01 6.13139503917700001700552547845291e-01 2.25711332013000011420533752470874e-02
            6.11262776677899966060181213833857e-01 1.30036083460900009356464579468593e-01 2.28570785871250008181032598031379e-02
            1.30518213593399989846943753946107e-01 2.58171382888400025024111528182402e-01 2.29069766352250001051338301749638e-02
            4.28143799182799988933822987746680e-01 2.36200596981700011722793419721711e-01 2.56393343723999994621109976833395e-02
            3.35699578373000018238769825984491e-01 4.31102630858800017765730672181235e-01 2.58289915353249986684769368139314e-02
            2.30542429883599991891074409977591e-01 3.45601394937600003842703699774574e-01 2.58963591798249993802816248944509e-02
            ];
        
    case  14
        
        % ALG. DEG.:   14
        % PTS CARD.:   45
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.5e-13
        
        xyw=[
            0.00000000000000000000000000000000e+00 1.00000000000000000000000000000000e+00 2.65417799749999999431998576326919e-04
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 2.65417799749999999431998576326919e-04
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 2.65417799749999999431998576326919e-04
            5.73330873025999984693434896598774e-02 1.51382269814000006147303167836071e-02 3.28650590252499994262103122366625e-03
            5.73330873025999984693434896598774e-02 9.27528685716000000915926193556515e-01 3.28650590252499994262103122366625e-03
            9.27528685716000000915926193556515e-01 5.73330873025999984693434896598774e-02 3.28650590252499994262103122366625e-03
            1.51382269814000006147303167836071e-02 5.73330873025999984693434896598774e-02 3.28650590252499994262103122366625e-03
            9.27528685716000000915926193556515e-01 1.51382269814000006147303167836071e-02 3.28650590252499994262103122366625e-03
            1.51382269814000006147303167836071e-02 9.27528685716000000915926193556515e-01 3.28650590252499994262103122366625e-03
            8.15962504071100047653430920036044e-01 1.65971996956500000841572273202473e-01 6.07204817372499969230226213312562e-03
            8.15962504071100047653430920036044e-01 1.80654989724000000772541341120814e-02 6.07204817372499969230226213312562e-03
            1.65971996956500000841572273202473e-01 8.15962504071100047653430920036044e-01 6.07204817372499969230226213312562e-03
            1.80654989724000000772541341120814e-02 8.15962504071100047653430920036044e-01 6.07204817372499969230226213312562e-03
            1.65971996956500000841572273202473e-01 1.80654989724000000772541341120814e-02 6.07204817372499969230226213312562e-03
            1.80654989724000000772541341120814e-02 1.65971996956500000841572273202473e-01 6.07204817372499969230226213312562e-03
            3.16547555637800015482241633435478e-01 1.86886898773000013906919747341817e-02 7.91999665829999990873666604329628e-03
            6.64763754484899993535407247691182e-01 1.86886898773000013906919747341817e-02 7.91999665829999990873666604329628e-03
            1.86886898773000013906919747341817e-02 6.64763754484899993535407247691182e-01 7.91999665829999990873666604329628e-03
            1.86886898773000013906919747341817e-02 3.16547555637800015482241633435478e-01 7.91999665829999990873666604329628e-03
            3.16547555637800015482241633435478e-01 6.64763754484899993535407247691182e-01 7.91999665829999990873666604329628e-03
            6.64763754484899993535407247691182e-01 3.16547555637800015482241633435478e-01 7.91999665829999990873666604329628e-03
            1.92662192491999985366479819504093e-02 4.90366890375399988588611677187146e-01 8.73294867589999995738381244336779e-03
            4.90366890375399988588611677187146e-01 1.92662192491999985366479819504093e-02 8.73294867589999995738381244336779e-03
            4.90366890375399988588611677187146e-01 4.90366890375399988588611677187146e-01 8.73294867589999995738381244336779e-03
            8.75134669581000002036574869634933e-02 8.24973066083700024009317530726548e-01 9.59161334862499923015199954079435e-03
            8.75134669581000002036574869634933e-02 8.75134669581000002036574869634933e-02 9.59161334862499923015199954079435e-03
            8.24973066083700024009317530726548e-01 8.75134669581000002036574869634933e-02 9.59161334862499923015199954079435e-03
            9.35526036219000017846170180746412e-02 2.07986542316699996035822550766170e-01 1.44592372802500000739112095970995e-02
            9.35526036219000017846170180746412e-02 6.98460854061299984962829512369353e-01 1.44592372802500000739112095970995e-02
            2.07986542316699996035822550766170e-01 9.35526036219000017846170180746412e-02 1.44592372802500000739112095970995e-02
            6.98460854061299984962829512369353e-01 9.35526036219000017846170180746412e-02 1.44592372802500000739112095970995e-02
            6.98460854061299984962829512369353e-01 2.07986542316699996035822550766170e-01 1.44592372802500000739112095970995e-02
            2.07986542316699996035822550766170e-01 6.98460854061299984962829512369353e-01 1.44592372802500000739112095970995e-02
            9.74892983467000040498362523067044e-02 5.38008859514899961951073237287346e-01 1.81455421848500005055360162486977e-02
            3.64501842138300002904571783801657e-01 9.74892983467000040498362523067044e-02 1.81455421848500005055360162486977e-02
            5.38008859514899961951073237287346e-01 9.74892983467000040498362523067044e-02 1.81455421848500005055360162486977e-02
            5.38008859514899961951073237287346e-01 3.64501842138300002904571783801657e-01 1.81455421848500005055360162486977e-02
            3.64501842138300002904571783801657e-01 5.38008859514899961951073237287346e-01 1.81455421848500005055360162486977e-02
            9.74892983467000040498362523067044e-02 3.64501842138300002904571783801657e-01 1.81455421848500005055360162486977e-02
            2.21714589487299995607116898099775e-01 5.56570821025299977691247477196157e-01 2.24464131026749988406798763662664e-02
            5.56570821025299977691247477196157e-01 2.21714589487299995607116898099775e-01 2.24464131026749988406798763662664e-02
            2.21714589487299995607116898099775e-01 2.21714589487299995607116898099775e-01 2.24464131026749988406798763662664e-02
            3.86047166929600005236267179498100e-01 2.27905666140799989527465641003801e-01 2.58636133404250012635117172976607e-02
            2.27905666140799989527465641003801e-01 3.86047166929600005236267179498100e-01 2.58636133404250012635117172976607e-02
            3.86047166929600005236267179498100e-01 3.86047166929600005236267179498100e-01 2.58636133404250012635117172976607e-02
            ];
        
    case  16
        
        % ALG. DEG.:   16
        % PTS CARD.:   55
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 7.8e-13
        
        xyw=[
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 1.55064996275000002885768868310379e-04
            0.00000000000000000000000000000000e+00 1.00000000000000000000000000000000e+00 1.57879367799999994039297779302444e-04
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 1.77165038975000000819742806967838e-04
            9.39886358357699958077091650920920e-01 4.98487446340000004557646562375339e-03 1.37909290419999997802380509170916e-03
            5.43806683058000034369783293186629e-02 9.38640561861699973533745833265129e-01 1.56731019139999994300993524376509e-03
            9.39400491640000022408596436207517e-03 5.26424462697000009048231561337161e-02 1.96328522064999995075385008647118e-03
            1.64345086361999993818105991749690e-02 9.46903551735099990871447062090738e-01 2.36378709660000014061709272539247e-03
            9.46948726986200028932216810062528e-01 3.63373677167000003196406510141969e-02 2.44561278177499999073041259123329e-03
            4.26604005767999977516424792156613e-02 1.51224541798999998626351626285214e-02 2.49654108722499991704357746868936e-03
            1.22269495439000005360119516240047e-02 8.69377351066400017032265168381855e-01 3.43884547040000003795268135320384e-03
            8.67369652104699961903122584772063e-01 1.20491728577399995203656146713911e-01 3.52447945099999999929440974710815e-03
            8.45674402138899994341159072064329e-01 1.57763967870000014481757233397730e-02 3.74117160842499978448683073395387e-03
            1.39575963210299996974939062965859e-01 8.44812087037499992447919794358313e-01 3.90243759030000020651685055383950e-03
            1.31782174323099993129559948101814e-01 1.35009605583999992006738466443494e-02 3.94209233370000026314139418559535e-03
            1.57955126300000012196100840355939e-02 1.45527493853600004003467915936199e-01 4.39486365957500024065351951207958e-03
            7.36546288443600039030911830195691e-01 1.55697540907999993492483170598462e-02 5.10284600675000015596038949183821e-03
            1.39688430330000001866608272393933e-02 7.37983689445000035078692235401832e-01 5.23907196540000005130499616257111e-03
            2.54789518603899978188565000891685e-01 7.29761568977099983612788491882384e-01 5.26783532495000029333143132248551e-03
            7.31638652255499954968342990468955e-01 2.54307668331499980851617692678701e-01 5.44116900505000038873459899946283e-03
            1.57253728950999989433245929149052e-02 2.69623979579100003789449147006962e-01 5.55721021747499974163808644789242e-03
            2.66230284364699976151769078569487e-01 1.44783956308000005819280886498746e-02 5.60466734205000011170838547513995e-03
            8.67350406521400052461956420302158e-01 5.91679410399999980429974755224976e-02 5.75306542482499974622278671176900e-03
            7.41493666957000058248539176020131e-02 8.63478257506099966533952283498365e-01 5.92034756249999998684030444451309e-03
            1.59285948359999991741364766539846e-02 4.19123895523800027440586291049840e-01 6.43661608420000030950092195780599e-03
            1.56061028068000002405613102496318e-02 5.80922292114600002932434108515736e-01 6.44892004019999992570078006792755e-03
            5.91009481748399956302364444127306e-01 1.59251452650999991700331293031923e-02 6.45180819024999963051669737978955e-03
            4.03477149688900027513227541930974e-01 5.80670036810399969873230929806596e-01 6.50858080147500000361970862172711e-03
            5.69474562852600008078240989561891e-01 4.14949514630200022580197583010886e-01 6.64420354022499971835546972442899e-03
            6.78493700649999975649251382492366e-02 7.61218678590999953126328136931988e-02 6.64461904577500015595115101518786e-03
            4.26596859027200014935488070477732e-01 1.57509692312000001979122032480518e-02 6.68830823095000002315169140842954e-03
            6.70982507890000023920151761558373e-02 7.74189831242099946173595981235849e-01 9.39469516602500082091964372921211e-03
            7.52831023147999989220124916755594e-01 8.19119495638999939313151799069601e-02 9.57664735487499980959302803285027e-03
            7.75372778355699954211388558178442e-01 1.57712845729199996336333811086661e-01 9.62124237562500018994260386762107e-03
            1.68907315778699995689748902805150e-01 7.50394309974199980040054924756987e-01 9.74049564629999968523588194102558e-03
            1.68733583291899996714491294369509e-01 7.08311507268000045689504418078286e-02 9.86510278867499916521932590285360e-03
            8.21244708436000003803911795330350e-02 1.76299662677100010821362729984685e-01 1.03091194524499996099109466740629e-02
            6.28870536334499963260213917237706e-01 8.07744953317000025094785087276250e-02 1.28218109620750007199285747105932e-02
            8.11413015265999987279954552832351e-02 3.05437358977599993092866270671948e-01 1.29101410483750001639835502942333e-02
            2.96911206508000014014925227456843e-01 6.22748598887099991117111130733974e-01 1.29557510567249994232197352062030e-02
            7.67542314170999995237565372008248e-02 6.24724714954600024441333516733721e-01 1.32131997045249992284166040690252e-02
            6.22302233384499969837122534954688e-01 3.01148582116600005420536945166532e-01 1.34626393256749993243559515576635e-02
            3.10378628805099976428749641854665e-01 7.79098365078999960386951784130360e-02 1.35473832329750000502688322967515e-02
            8.19218215187000053933985554976971e-02 4.60363303835099990646284595641191e-01 1.46184286611000004152405296053985e-02
            4.71702266501299982692785306426231e-01 8.21554006796999941597547945093538e-02 1.48215792090750000248311479822405e-02
            4.54660341525000011309032288409071e-01 4.63756503388999985482143983972492e-01 1.48589569187250001430777857081011e-02
            1.70109133923700001078671562027012e-01 6.42227780818800053630468482879223e-01 1.57950063965749988581510621088455e-02
            6.40600432948699949342596937640337e-01 1.89829353725600002977813574034371e-01 1.58231711288249997371391231126836e-02
            1.91226758371700000793680374044925e-01 1.73995568534300010243853762403887e-01 1.60176840442999984803762458795973e-02
            1.88531576706999992598667859056150e-01 4.79891407040600015410092282763799e-01 2.03010148979500017196730965451934e-02
            4.77292995769100003577989355108002e-01 3.34835659811899999382944770331960e-01 2.03609378382499997539589742245880e-02
            3.12697462175999973954532151765306e-01 4.95797219725900017994746349359048e-01 2.03669800310249983488208869175651e-02
            4.96122594594600008210960595533834e-01 1.92755366890399987278570392845722e-01 2.03762637020999982928604055132382e-02
            1.92880531286699991033728451839124e-01 3.16101580726100006568657363459351e-01 2.03791166234749999974784628875568e-02
            3.36004145381600005659805674440577e-01 1.89489280128999987296012363913178e-01 2.04232764905750013850749269295193e-02
            3.33728055084800023788460521245725e-01 3.34357102181099985482859437979641e-01 2.30804583632749985011578530702536e-02
            ];
        
    case  18
        
        % ALG. DEG.:   18
        % PTS CARD.:   66
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.0e-13
        
        xyw=[
            1.16731059668000007600863199286323e-02 9.81256595128899955504664376348956e-01 6.29143924649999989814896927953214e-04
            9.81003085838800026330375203542644e-01 7.14625048630000009319829246123845e-03 6.31836300174999968524980342721165e-04
            1.06966317092000006566676617580924e-02 1.15153933376000005422978844649151e-02 8.31732383324999954496425935701609e-04
            9.38247698355100023803743169992231e-01 4.95570591341000024621088471121766e-02 2.03758730312499991912944530270124e-03
            1.26627518416999994971261500609216e-02 9.37012362061499959331456466316013e-01 2.15338814355000001538043008508794e-03
            5.98109409984000015003680061909108e-02 1.21364578921999994448599480278972e-02 2.19466865447500011562786959018467e-03
            1.37363297926999996384500590806965e-02 6.12783625596999975693179862901161e-02 2.42748963905000010587986736254607e-03
            9.22952795940499948201818369852845e-01 1.41128270601999991629149633354245e-02 2.56165529787500019509582926957592e-03
            6.33107354992999971088352140213829e-02 9.22019729172700031050169400259620e-01 2.70994220852500012491903547129368e-03
            1.17265100335000004017071972839403e-02 1.50052047522899995035672304766194e-01 3.23463475440000005484142420186799e-03
            1.55472058732299994732173331613012e-01 8.32514712158900049843168744700961e-01 3.40849558957500003292251378184119e-03
            8.34329388898199963975343962374609e-01 1.25228158758999997407501680868336e-02 3.46193320367499998008176120833923e-03
            8.50163803195699951942287952988409e-01 1.37199750873600007095021169334359e-01 3.48553850262500004505983319802453e-03
            1.28816350522000001854738826523317e-02 8.47762706347899963610359463928035e-01 3.60303499919999983477914007323761e-03
            1.51080160895900011963632891820453e-01 1.36526924038999995097976025704156e-02 3.84258638834999999289721017703414e-03
            1.01917879217000000718895691420585e-02 5.77043861834499960394850859302096e-01 4.06224505632500013790897597232288e-03
            2.81337239930300020418485473783221e-01 7.06685375962299966445812060555909e-01 4.24295760700000006843124467081907e-03
            7.12437462850100011735321459127590e-01 1.24569780989999996656525382832115e-02 4.25221331052500015751371975625261e-03
            2.76302525086299988199556310064509e-01 1.21741311386000007621621676889845e-02 4.27383801687499995103358685355488e-03
            1.09658368560999995577898147303131e-02 4.19430671246599995338755206830683e-01 4.34722136397500038412777811913656e-03
            4.28911051788399999562528819296858e-01 5.59961606746899986930543491325807e-01 4.36359906097500034605207730464826e-03
            4.21542055511499980102740892107249e-01 1.16475994784999998393759668147140e-02 4.46016893217499998253172677209477e-03
            5.71125859044400052688672531076008e-01 1.18218313988999994962325601477460e-02 4.46117159697499970211209685544418e-03
            5.82686827051099975882664239179576e-01 4.05788958117699993799476487765787e-01 4.47615843879999985438500331724754e-03
            1.30567806713000004320557323467256e-02 2.72502375086799975001383700146107e-01 4.53149390502500040711630902023899e-03
            1.30760400963999996737374686972544e-02 7.22471252323300006104034309828421e-01 4.61962097205000001703956158394249e-03
            7.26343706240699993337273099314189e-01 2.60298401925100009179914195556194e-01 4.64483910927500007193025055585167e-03
            6.87230068637000035547401921576238e-02 6.31417277210000044140869590592047e-02 5.08042879442500028980278514723068e-03
            8.65230210152900025555311458447250e-01 7.20611837337999938668531285657082e-02 5.34429154522499984258931959857364e-03
            6.48599071037000013406625953393814e-02 8.59043354390999991387900536210509e-01 5.79792135244999989190928246785006e-03
            1.48349494336200010691229067560926e-01 7.88878835224000041215219880541554e-01 6.86066777147500018313452940788011e-03
            6.24359898395999979303638838246115e-02 1.49393549935399994055273964477237e-01 7.25754805849999993366816752882187e-03
            7.87136901173499992090398791333428e-01 6.56382042756999961596875436953269e-02 7.36306846262500033079012595749191e-03
            5.19104921610000005771645703589456e-02 5.25563569560499987964874435419915e-01 7.48590629072500013396762597039924e-03
            1.54312992744399996025705945612572e-01 7.16383926916999980161193661842844e-02 7.67567370297500030373871737765512e-03
            2.61784274560300012080205078746076e-01 6.21479485287999983134099579729082e-02 8.13158414657499939437457214808092e-03
            7.66725787281300030784336740907747e-01 1.65821155483100002303586961716064e-01 8.19710521264999970869613576951451e-03
            2.58210367662700002178155500587309e-01 6.80011976613900004373647334432462e-01 8.28086687979999978848066888303947e-03
            6.79065925147000065686953007570992e-02 7.57151543778200042389414647914236e-01 8.65418817187500050147619390372711e-03
            5.29357827480399945052624843810918e-01 4.12150384110699985473758033549530e-01 8.67703434940000038755325562078724e-03
            6.66036150483999972449211668390490e-02 2.61251308788599978516486999069457e-01 8.68430123510000040454315239912830e-03
            5.85675461898999966114942594685999e-02 3.90223611453499985568527108625858e-01 8.71321906134999921156314428571932e-03
            6.44535360411000041169771179738746e-02 6.37362655976099978794024991657352e-01 8.71503902965000028235653672936678e-03
            6.74813842915100048358567619288806e-01 6.37583342060999980205338033556473e-02 8.88678924937500065717177477608857e-03
            3.91460231036899974910170385555830e-01 5.50323809056299961817160237842472e-01 9.00457490957499995243296098124119e-03
            6.48770149230699977316305648855632e-01 2.83672836026300001499578229413601e-01 9.07315714607499941568580226203267e-03
            3.94649822040800013489558750734432e-01 6.05175522554000019925801723275072e-02 9.54744255207500003290732593086432e-03
            5.39013715193299991490505362889962e-01 6.11990176935999982688052512003196e-02 9.80632000294999993383893155396436e-03
            1.62789508278500000271193925982516e-01 6.86132214103500004398483724799007e-01 1.20677531471999992251920019725731e-02
            6.81243632264100007489560084650293e-01 1.56796834589900013767760356131475e-01 1.22478030391500007645344538786958e-02
            1.54283287802000013133607581039541e-01 1.66751262402000000051316419558134e-01 1.24305208468000003702025324514580e-02
            2.52272775044500019436810589468223e-01 2.50480393339500007066078524076147e-01 1.26766434246500005189606241629008e-02
            2.54798153240699987165385209664237e-01 4.99409064904299981879631786796381e-01 1.27442998510749992691071241779355e-02
            1.48558054919399995519313506520120e-01 5.75602309608699957976796213188209e-01 1.30340015916750005808832924003582e-02
            2.93023960643599978848072851178586e-01 5.65689735416200045747814328933600e-01 1.30865218731249996486631914649479e-02
            2.80899127231000023119378283809056e-01 1.43792157424800004505982542468701e-01 1.31110170887999994554506955068973e-02
            4.82098959297100004839364828512771e-01 2.51855753586499975238410797828692e-01 1.31864911205749994749236719826513e-02
            5.64187824544399973980546292295912e-01 1.46296674315299995994266168963804e-01 1.32362265931999997792267720342352e-02
            1.30769964434400004504865933085966e-01 4.48957758611700008533063055438106e-01 1.35598898625249996569008814617519e-02
            1.47969222194799993053138109644351e-01 3.00117438682899972945961053483188e-01 1.35867550854749993133330221439792e-02
            5.63868422294599991317909370991401e-01 2.81377208929800004533205992629519e-01 1.36775137159750002913138899884871e-02
            4.36115742879000012344192782620667e-01 4.25205344641999993804404311958933e-01 1.39322086478249999863532337940342e-02
            3.60326393528499988594404612740618e-01 2.59919000488899987200142049914575e-01 1.44433566058250004854324188841019e-02
            4.22418833467400001158154054792249e-01 1.45323844330299989380250735848676e-01 1.46348445405750007125567080379369e-02
            3.71900183305200027206183222006075e-01 3.78012270356699986439963367956807e-01 1.52259812669999997775116540310592e-02
            2.41364500692799993641912692510232e-01 3.84756328493999988360485531302402e-01 1.59318491112249993602034692230518e-02
            ];
        
    case  20
        
        % ALG. DEG.:   20
        % PTS CARD.:   78
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 7.7e-13
        
        xyw=[
            8.94113371120000034741881478339565e-03 8.69832937020000029892941029174835e-03 5.43613634974999962792829855118271e-04
            9.79262262980699960124297831498552e-01 1.02644133744000005947771825276504e-02 7.24678381625000023018301487098825e-04
            1.05475382112000005963015070165056e-02 9.78551420251499992630783708591480e-01 7.71150733424999946380440363213893e-04
            2.37770619469999987607988778393064e-03 6.36551098604000026304206016902754e-02 8.60040827599999990575596608977094e-04
            6.30425115794999990637137443627580e-02 4.15063475090000043310478616831460e-03 1.04746180030000006315471505757841e-03
            9.30842249672999999532407855440397e-01 4.80534822629999965742930712053749e-03 1.11845128744999990771935571842732e-03
            6.29076555490000050019361310660315e-02 9.31679006948100019158687246090267e-01 1.17636052035000005996923100326512e-03
            9.31596224638099990045247977832332e-01 6.26264881800999984040601020751637e-02 1.22169839375000006433114041470844e-03
            6.19516894150000038687942094384198e-03 9.29358705856400013622931055579102e-01 1.29819108422499999294652539560957e-03
            2.87125819236999993422365662354423e-02 3.10202122996999993254085836724698e-02 1.85182647452500001583608479904797e-03
            9.29384447830500004528175850282423e-01 3.42152968218999986582673500379315e-02 1.99388525752499995088995454750602e-03
            3.75457566620999994011498301915708e-02 9.25786888466900048655361388227902e-01 2.08876307275000002577902691314193e-03
            8.68957390639999924242253825923399e-03 1.58497125151000006759716143278638e-01 2.40416652159999985269300815104998e-03
            1.54759705396499996910009144812648e-01 8.36360665768799971253599778719945e-01 2.40795644624999978511414866488849e-03
            8.33102529418500048485896058991784e-01 8.92572448239999961805413164483980e-03 2.46443651895000013021808094038079e-03
            8.37423107352599949138038937235251e-01 1.52916730407799988933348345199192e-01 2.56644700752500006663003873086382e-03
            1.55936250523399988487938117032172e-01 9.49662400579999975425149472130215e-03 2.57970257777499984222502327213533e-03
            9.85996420949999952121345359046245e-03 8.34221149359600011585769152588909e-01 2.65727504074999990382743852990188e-03
            4.05587373328899991609119979329989e-01 7.43893020079999973959150594282619e-03 2.67203267237499982739157289302057e-03
            5.96472789861799967958688739599893e-01 3.95633080931100000743327882446465e-01 2.67422552524999987064169282291459e-03
            8.07478004159999963873506345635178e-03 4.03131942590299996442837482391042e-01 2.72566154285000007695405344065875e-03
            7.50739777210000010926327362881239e-03 5.85160959468099983737943148298655e-01 2.74749458937499982380536778237001e-03
            3.93676451923699988189042642261484e-01 5.97489659289900054695010567229474e-01 2.83557638072499994599806605322101e-03
            5.84653072621200031733224022900686e-01 8.72504649680000063383111807979731e-03 3.01339107325000006218362891274865e-03
            4.87080411212000019549606122382102e-01 2.02129229911999983015924442497635e-02 3.49047984552500016597331722323361e-03
            2.68351281178499989632513234028011e-01 7.20234008866800001769092887116130e-01 3.52869978839999995715026592790764e-03
            7.22395628874800022067859117669286e-01 2.66239936645599972742104455392109e-01 3.54825867615000001970004817053450e-03
            2.71682674235699972431490323288017e-01 1.12882698808000007245633256047768e-02 3.60531690670000013951290718239306e-03
            1.12580842045999999362360100008118e-02 7.16969596332500014312927305581979e-01 3.61760867137499988674398387900055e-03
            1.15034734370000005926337749428967e-02 2.74006711016600013053334805590566e-01 3.62374424680000006490643293943776e-03
            7.14052590056399982820778404857265e-01 1.13511560497000004582579535394871e-02 3.63466939234999980579132738967019e-03
            4.90287105311200011037442436645506e-01 4.93649184146799979711062178466818e-01 3.64910477314999998760258748120577e-03
            2.01423425209000007962689693385983e-02 4.83257345960100004322868016970460e-01 3.68286446165000001459755019084241e-03
            3.61107464858999999135491520974028e-02 9.35679501582000056059129633467819e-02 4.18659908259999980623122795009294e-03
            8.60799881985100046044578903092770e-01 3.97379067074999994813033765694854e-02 4.22388751145000036074650395789831e-03
            1.00589152600099998591254291113728e-01 8.58634341935200051665333376149647e-01 4.23556657209999982277581764833485e-03
            9.18740717058000033734543876562384e-02 3.95513001973000002942093544788804e-02 4.32675430237500040220322006234710e-03
            8.60488829619100004109100154892076e-01 9.66224057078999970515909012647171e-02 4.36311366232499959344481865741727e-03
            4.39842178673000017141703210654669e-02 8.56188634910699963853630833909847e-01 4.43043055397500037950742068915133e-03
            2.01101760673499996290303215573658e-01 7.44911583562600010388621285528643e-01 7.07060060057500012337117212268822e-03
            7.44999372626300004895938400295563e-01 5.36865638166000025410795615243842e-02 7.12491781219999967267320073460724e-03
            5.32186641310000013715963973481848e-02 1.96375427593499990441827662834839e-01 7.12514116347499979614577014785937e-03
            7.45398464740099964132014065398835e-01 1.98206580555000011489852340673679e-01 7.51618058694999979785222876671469e-03
            1.95728993287599989381320142456389e-01 5.55713833155999994972695787964767e-02 7.55078192704999959922185226446345e-03
            1.09253205798800004378001915483765e-01 6.10003618241300005209382106841076e-01 7.59967840192499969470540932547920e-03
            5.67625702000999973750872129585332e-02 7.40912189495899986546589843783295e-01 7.64171990185000000322945723496559e-03
            4.83837933475000023242351687713381e-02 6.07513566097800050691546402958920e-01 7.65168532505000002080786103419996e-03
            1.08061280975999998710257443690352e-01 1.12208151043699994042768253166287e-01 7.73325170502500019997782487735094e-03
            6.18560590099100027217104980081785e-01 2.69875370303500006219366014192929e-01 7.74434552087499995104424854730496e-03
            7.72129601349699989931707477808231e-01 1.11411739533299997129134339957091e-01 7.82865626362500011126144272566307e-03
            6.11573480113300038674140068906127e-01 3.38936767793100002243988910777261e-01 7.83933733480000026483391906140241e-03
            3.38132610337600003624913824751275e-01 4.94693938787000020429474034244777e-02 7.85801173217499926426654610622791e-03
            1.17308412825399999146469554034411e-01 7.69645130979500047274655116780195e-01 7.87955359735000027132656441608560e-03
            2.67455126059599979093661659135250e-01 1.11571880815399995801051602484222e-01 8.10620344962499975172143251711532e-03
            6.54210016002599981099763226666255e-01 1.90654831470000007964671340232599e-01 8.68780380965000070814241439620673e-03
            5.38297481157999982426254348411021e-02 3.35861682684900020312568358349381e-01 8.75983637317499959362709915922096e-03
            1.84884032411700011122590581180702e-01 1.55183152385099998316064784376067e-01 8.76793550774999930208508658324718e-03
            3.37626710474400015815632514204481e-01 6.08140259629399992924447815312305e-01 8.80323038334999974363626051854226e-03
            6.06710203449900009609052631276427e-01 5.42632795598000025605323060062801e-02 8.81538762452500034016988905705148e-03
            4.61261408549600016026914772737655e-01 6.88176670721999950153247027628822e-02 9.16008050857500082109208250358279e-03
            1.52546536567099988435103341544163e-01 6.51024084574900019362075909157284e-01 9.19332769175000043271328564742362e-03
            7.00582543543000063612780081712117e-02 4.66190439274200008057391642068978e-01 9.29189157342500023084586047161793e-03
            4.70420137903199975237811258921283e-01 4.63482645535299997430200846793014e-01 9.33428929015000033442284177453985e-03
            1.21646169374600002233144380170415e-01 2.38149487551600008705676714271249e-01 1.00993336647000001710106076302509e-02
            6.37140405270199994269830767734675e-01 1.23839938451299994004450866214029e-01 1.03395010159499992075105012645508e-02
            2.37990451511899986369868997826416e-01 6.37021645232599964536746028898051e-01 1.05489447967500003799923291580853e-02
            1.48392985717699987802120631386060e-01 4.89418857777999982694439040642465e-01 1.23862751009249996703154295119020e-02
            3.59806957154999995651678545982577e-01 1.45288086625299989185222671039810e-01 1.25104815285250001860584134760757e-02
            4.94144105509499997896938339181361e-01 3.61021638381800014094835660216631e-01 1.26448646778749996394397570043111e-02
            1.44063068798099991330019520319183e-01 3.51350834188700000826344194138073e-01 1.30009302546999997579613150833211e-02
            5.01976444000399979472604172769934e-01 1.43549166329300009747882427291188e-01 1.30383391971500006700779294988024e-02
            3.55542383429799990768316320099984e-01 5.01649159950199985402718994009774e-01 1.31224788089500006560417233458793e-02
            2.44343954077099995148003586109553e-01 2.40605212910399990366983047351823e-01 1.49789940629000001282022935811256e-02
            2.43706498934199988770643585667131e-01 5.10901727705500019460771454760106e-01 1.49902499356500008148573854782626e-02
            5.12220080732100013776175728708040e-01 2.45273797354299993855875072767958e-01 1.49978818032249997665994456497174e-02
            2.52603831517800025974196387323900e-01 3.70031955509399990145169567767880e-01 1.58533295862249984375935696334636e-02
            3.75989565285100024727427125981194e-01 2.50540661163099975805579333609785e-01 1.58827965276999993893269902400789e-02
            3.72907798714399996153900929130032e-01 3.75375027754900003618132586780121e-01 1.59301651417999984294837645393272e-02
            ];
        
    case  21
        
        % ALG. DEG.:   21
        % PTS CARD.:   91
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 8.9e-13
        
        xyw=[
            3.55243919219999995662284675290721e-03 3.55243919219999995662284675290721e-03 1.67610910975000000329784843877690e-04
            3.55243919219999995662284675290721e-03 9.92895121615600007025648210401414e-01 1.67610910975000000329784843877690e-04
            9.92895121615600007025648210401414e-01 3.55243919219999995662284675290721e-03 1.67610910975000000329784843877690e-04
            9.55354827373000037304962006601272e-01 8.78989290930000041757175210932473e-03 1.13681520185000004344311186343930e-03
            3.58552797177000004413827127791592e-02 8.78989290930000041757175210932473e-03 1.13681520185000004344311186343930e-03
            9.55354827373000037304962006601272e-01 3.58552797177000004413827127791592e-02 1.13681520185000004344311186343930e-03
            8.78989290930000041757175210932473e-03 3.58552797177000004413827127791592e-02 1.13681520185000004344311186343930e-03
            8.78989290930000041757175210932473e-03 9.55354827373000037304962006601272e-01 1.13681520185000004344311186343930e-03
            3.58552797177000004413827127791592e-02 9.55354827373000037304962006601272e-01 1.13681520185000004344311186343930e-03
            8.86526487904700011277725479885703e-01 1.08232974501699996672243742068531e-01 1.30193963300000007700385484810113e-03
            8.86526487904700011277725479885703e-01 5.24053759349999998679026091963351e-03 1.30193963300000007700385484810113e-03
            5.24053759349999998679026091963351e-03 1.08232974501699996672243742068531e-01 1.30193963300000007700385484810113e-03
            5.24053759349999998679026091963351e-03 8.86526487904700011277725479885703e-01 1.30193963300000007700385484810113e-03
            1.08232974501699996672243742068531e-01 8.86526487904700011277725479885703e-01 1.30193963300000007700385484810113e-03
            1.08232974501699996672243742068531e-01 5.24053759349999998679026091963351e-03 1.30193963300000007700385484810113e-03
            4.66397432150000013084678585073561e-02 9.06720513570000052894215514243115e-01 1.63588582217500007331967282198093e-03
            4.66397432150000013084678585073561e-02 4.66397432150000013084678585073561e-02 1.63588582217500007331967282198093e-03
            9.06720513570000052894215514243115e-01 4.66397432150000013084678585073561e-02 1.63588582217500007331967282198093e-03
            2.07572045694599999210439023045183e-01 8.27592412839999944096369688395498e-03 2.31844603832499990617166396589255e-03
            2.07572045694599999210439023045183e-01 7.84152030176999970123574712488335e-01 2.31844603832499990617166396589255e-03
            7.84152030176999970123574712488335e-01 2.07572045694599999210439023045183e-01 2.31844603832499990617166396589255e-03
            8.27592412839999944096369688395498e-03 7.84152030176999970123574712488335e-01 2.31844603832499990617166396589255e-03
            8.27592412839999944096369688395498e-03 2.07572045694599999210439023045183e-01 2.31844603832499990617166396589255e-03
            7.84152030176999970123574712488335e-01 8.27592412839999944096369688395498e-03 2.31844603832499990617166396589255e-03
            8.58119489724999962021456667571329e-02 3.14836947700999972288293804467685e-02 2.39844456557500003771044028155757e-03
            8.82704356257399957996767625445500e-01 3.14836947700999972288293804467685e-02 2.39844456557500003771044028155757e-03
            3.14836947700999972288293804467685e-02 8.58119489724999962021456667571329e-02 2.39844456557500003771044028155757e-03
            8.58119489724999962021456667571329e-02 8.82704356257399957996767625445500e-01 2.39844456557500003771044028155757e-03
            8.82704356257399957996767625445500e-01 8.58119489724999962021456667571329e-02 2.39844456557500003771044028155757e-03
            3.14836947700999972288293804467685e-02 8.82704356257399957996767625445500e-01 2.39844456557500003771044028155757e-03
            6.68877823382600045221124673844315e-01 9.51507606250000075032158264320969e-03 2.85619522917499989861367382104618e-03
            9.51507606250000075032158264320969e-03 3.21607100555000025021712417583331e-01 2.85619522917499989861367382104618e-03
            9.51507606250000075032158264320969e-03 6.68877823382600045221124673844315e-01 2.85619522917499989861367382104618e-03
            6.68877823382600045221124673844315e-01 3.21607100555000025021712417583331e-01 2.85619522917499989861367382104618e-03
            3.21607100555000025021712417583331e-01 6.68877823382600045221124673844315e-01 2.85619522917499989861367382104618e-03
            3.21607100555000025021712417583331e-01 9.51507606250000075032158264320969e-03 2.85619522917499989861367382104618e-03
            4.37999954311300010445506813994143e-01 9.98597856810000059080145717871346e-03 2.93042410435000004464978218265969e-03
            9.98597856810000059080145717871346e-03 5.52014067120599971616456969059072e-01 2.93042410435000004464978218265969e-03
            4.37999954311300010445506813994143e-01 5.52014067120599971616456969059072e-01 2.93042410435000004464978218265969e-03
            9.98597856810000059080145717871346e-03 4.37999954311300010445506813994143e-01 2.93042410435000004464978218265969e-03
            5.52014067120599971616456969059072e-01 4.37999954311300010445506813994143e-01 2.93042410435000004464978218265969e-03
            5.52014067120599971616456969059072e-01 9.98597856810000059080145717871346e-03 2.93042410435000004464978218265969e-03
            7.97493107214800045490221691579791e-01 4.05093994119000017084530895772332e-02 4.70492888080000035838645899843868e-03
            4.05093994119000017084530895772332e-02 1.61997493373399997773631753261725e-01 4.70492888080000035838645899843868e-03
            4.05093994119000017084530895772332e-02 7.97493107214800045490221691579791e-01 4.70492888080000035838645899843868e-03
            1.61997493373399997773631753261725e-01 7.97493107214800045490221691579791e-01 4.70492888080000035838645899843868e-03
            7.97493107214800045490221691579791e-01 1.61997493373399997773631753261725e-01 4.70492888080000035838645899843868e-03
            1.61997493373399997773631753261725e-01 4.05093994119000017084530895772332e-02 4.70492888080000035838645899843868e-03
            3.86421555195499977219952825180371e-01 3.86421555195499977219952825180371e-01 5.88152450677500038961342809784583e-03
            3.86421555195499977219952825180371e-01 2.27156889608999990048943118381430e-01 5.88152450677500038961342809784583e-03
            2.27156889608999990048943118381430e-01 3.86421555195499977219952825180371e-01 5.88152450677500038961342809784583e-03
            8.09012937932899967030664356570924e-01 9.54935310336000042763515693877707e-02 5.88928665377499993560661550873192e-03
            9.54935310336000042763515693877707e-02 8.09012937932899967030664356570924e-01 5.88928665377499993560661550873192e-03
            9.54935310336000042763515693877707e-02 9.54935310336000042763515693877707e-02 5.88928665377499993560661550873192e-03
            2.74542523871800003831822323263623e-01 4.79840480721000009478593995027040e-02 6.70615518574999968509642300773521e-03
            4.79840480721000009478593995027040e-02 6.77473428056100002159212181140902e-01 6.70615518574999968509642300773521e-03
            6.77473428056100002159212181140902e-01 4.79840480721000009478593995027040e-02 6.70615518574999968509642300773521e-03
            6.77473428056100002159212181140902e-01 2.74542523871800003831822323263623e-01 6.70615518574999968509642300773521e-03
            2.74542523871800003831822323263623e-01 6.77473428056100002159212181140902e-01 6.70615518574999968509642300773521e-03
            4.79840480721000009478593995027040e-02 2.74542523871800003831822323263623e-01 6.70615518574999968509642300773521e-03
            4.05347244666700023962846444192110e-01 5.42984962234399959335462426679442e-01 7.85724441947499921412578771651170e-03
            5.16677930989000028239033213139919e-02 4.05347244666700023962846444192110e-01 7.85724441947499921412578771651170e-03
            4.05347244666700023962846444192110e-01 5.16677930989000028239033213139919e-02 7.85724441947499921412578771651170e-03
            5.42984962234399959335462426679442e-01 5.16677930989000028239033213139919e-02 7.85724441947499921412578771651170e-03
            5.16677930989000028239033213139919e-02 5.42984962234399959335462426679442e-01 7.85724441947499921412578771651170e-03
            5.42984962234399959335462426679442e-01 4.05347244666700023962846444192110e-01 7.85724441947499921412578771651170e-03
            1.87773861553899995557870283846569e-01 1.06814826758799996664350828723400e-01 8.42990480397499963860497018686146e-03
            7.05411311687199948927684545196826e-01 1.87773861553899995557870283846569e-01 8.42990480397499963860497018686146e-03
            7.05411311687199948927684545196826e-01 1.06814826758799996664350828723400e-01 8.42990480397499963860497018686146e-03
            1.06814826758799996664350828723400e-01 7.05411311687199948927684545196826e-01 8.42990480397499963860497018686146e-03
            1.87773861553899995557870283846569e-01 7.05411311687199948927684545196826e-01 8.42990480397499963860497018686146e-03
            1.06814826758799996664350828723400e-01 1.87773861553899995557870283846569e-01 8.42990480397499963860497018686146e-03
            1.19505971200899993589494840762200e-01 3.05712299064299974915570601297077e-01 1.06936323553250001161529780802084e-02
            1.19505971200899993589494840762200e-01 5.74781729734800017617146750126267e-01 1.06936323553250001161529780802084e-02
            5.74781729734800017617146750126267e-01 1.19505971200899993589494840762200e-01 1.06936323553250001161529780802084e-02
            5.74781729734800017617146750126267e-01 3.05712299064299974915570601297077e-01 1.06936323553250001161529780802084e-02
            3.05712299064299974915570601297077e-01 5.74781729734800017617146750126267e-01 1.06936323553250001161529780802084e-02
            3.05712299064299974915570601297077e-01 1.19505971200899993589494840762200e-01 1.06936323553250001161529780802084e-02
            5.98124574336299952292961279454175e-01 2.00937712831900011645203107946145e-01 1.10284733184250003518167559946050e-02
            2.00937712831900011645203107946145e-01 5.98124574336299952292961279454175e-01 1.10284733184250003518167559946050e-02
            2.00937712831900011645203107946145e-01 2.00937712831900011645203107946145e-01 1.10284733184250003518167559946050e-02
            2.16077520000500011843058700833353e-01 3.12136025667299976671387184978812e-01 1.15367398671000001564301840062399e-02
            3.12136025667299976671387184978812e-01 2.16077520000500011843058700833353e-01 1.15367398671000001564301840062399e-02
            2.16077520000500011843058700833353e-01 4.71786454332099980391035387583543e-01 1.15367398671000001564301840062399e-02
            3.12136025667299976671387184978812e-01 4.71786454332099980391035387583543e-01 1.15367398671000001564301840062399e-02
            4.71786454332099980391035387583543e-01 3.12136025667299976671387184978812e-01 1.15367398671000001564301840062399e-02
            4.71786454332099980391035387583543e-01 2.16077520000500011843058700833353e-01 1.15367398671000001564301840062399e-02
            4.37657990384899997504675184245571e-01 4.37657990384899997504675184245571e-01 1.17288117155999998403315132122771e-02
            4.37657990384899997504675184245571e-01 1.24684019230299994451804934669781e-01 1.17288117155999998403315132122771e-02
            1.24684019230299994451804934669781e-01 4.37657990384899997504675184245571e-01 1.17288117155999998403315132122771e-02
            3.33333333333300008138877501551178e-01 3.33333333333300008138877501551178e-01 1.37799995086750007994291777890794e-02
            ];
        
    case  23
        
        % ALG. DEG.:   23
        % PTS CARD.:  105
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 7.0e-13
        
        xyw=[
            8.78093038360000001374405798060252e-03 9.90367643677199982477077355724759e-01 1.60957456524999997305239163303270e-04
            9.90367531421999958141100250941236e-01 8.78092162320000084452242816723810e-03 1.60960326899999993290751865160360e-04
            2.70292764499999995861378998540658e-03 3.35914404439000002788873189274454e-02 2.53368392750000020422623814297935e-04
            3.35909214524000021562244455708424e-02 2.70289467099999987723113470394765e-03 2.53368814399999983579675477685100e-04
            9.16750686059999986277091466035927e-03 9.16763530509999945150578781749573e-03 4.91998248375000038837279170422789e-04
            9.67556818255799977457343175046844e-01 8.47371766560000086454529366619681e-03 8.36682844600000015164442679349577e-04
            8.47372006879999924155999480035462e-03 9.67556943534499946935056868824176e-01 8.36683480199999947413136425211633e-04
            7.87819487920000081915272716059917e-03 6.76784943861999943592877571063582e-02 1.07183308437499994272834591413357e-03
            6.76785477699999998835167502875265e-02 7.87816592909999953431299246631170e-03 1.07183649712500007541993163329153e-03
            9.47026695504699977945506361720618e-01 4.42974541187000012132379822560324e-02 1.07509503429999999334076843382491e-03
            4.42974755679999970814364473881142e-02 9.47026667648700026802544016391039e-01 1.07509622745000001792226029806443e-03
            9.14424321488199964846899092663080e-01 8.17354551320000066727544663081062e-03 1.42336573012499995917468087469615e-03
            8.17354244589999986980721757845458e-03 9.14424423403100039564606049680151e-01 1.42336600335000009256991582162755e-03
            2.49745229274100005723013850911229e-01 3.83323243471999974829600432713050e-01 1.54109670037500001184282982791274e-03
            3.83323264605499980373082280493691e-01 2.49745126800500011343686423970212e-01 1.54111891045000007176291045851713e-03
            8.87685035355699980641475121956319e-01 1.03532880944600003125799503322924e-01 1.55036283977499998038007689160622e-03
            1.03532922829700002598229957584408e-01 8.87684993183999959676100388605846e-01 1.55036329880000006524454203571395e-03
            7.72559236180000042520932268530487e-03 1.40319099197399987932755038855248e-01 1.74090825735000002960628506798457e-03
            1.40319242510699992987355244622449e-01 7.72559346239999958361188703293010e-03 1.74090829605000004844106431534101e-03
            8.10459100965200018862333308788948e-01 1.80964252392600000174383012563339e-01 1.87665644299999994845662421028010e-03
            1.80964300371699998715158130835334e-01 8.10459051533400032241161170531996e-01 1.87665661412499996940628221864245e-03
            8.33076794868400027915811278944602e-01 8.30109396770000070031603200959580e-03 1.97686920847499987205497795628162e-03
            8.30109071260000039305104735376517e-03 8.33076854539199995741682869265787e-01 1.97686931212499991325493020610793e-03
            3.48407706146999993079838020548777e-02 3.48406969482000020144063512361754e-02 2.00883361557500006097654043912826e-03
            2.74028767960800001546317616885062e-01 7.17398184794799997909819921915187e-01 2.19908602684999992388226175421551e-03
            7.17398222477800051599672315205680e-01 2.74028730438599987806469471252058e-01 2.19908620280000015323462747574013e-03
            2.39497685823400002780303452709632e-01 8.18591822620000006405227566119720e-03 2.28260489290000017498227791179488e-03
            8.18591858450000084412856438120798e-03 2.39497556667700006727272921125405e-01 2.28260534027500006787336772617891e-03
            6.88361520749999986762412973462233e-03 4.84374089268700014354607219502213e-01 2.32054371877500002080418894934155e-03
            4.84374148569900020433465215319302e-01 6.88362329490000042497843679711877e-03 2.32054539155000013467278208167954e-03
            4.96076777274099989245570441198652e-01 4.96076752950699972899428757955320e-01 2.36249515445000014476906180505011e-03
            6.11293677624500042533384203125024e-01 3.80432369123899993823556542338338e-01 2.36568671210000015109331528151415e-03
            3.80432398034500018901837847806746e-01 6.11293646653299993509733667451655e-01 2.36568713234999988132001291774031e-03
            7.30389071352400054415454633272020e-01 8.39871797010000031846743695496116e-03 2.38889430712499982456531455454751e-03
            8.39871686390000066202432549289369e-03 7.30389089540700031655262591812061e-01 2.38889482107500000437894094318381e-03
            6.12852567561199990109344071242958e-01 7.54759796949999988563417829823265e-03 2.40347106219999983103829954700359e-03
            7.54759610370000014190061676799814e-03 6.12852548458200030445652828348102e-01 2.40347117065000011326691975455105e-03
            7.95253165130000057303494287452850e-03 3.55977382672099995630787816480733e-01 2.49978810529999980610815768500288e-03
            3.55977487046000018455771396475029e-01 7.95253585020000083871671847646212e-03 2.49978879625000014602642650629605e-03
            9.11023697796599973841580322186928e-01 4.37233665344999994362140682824247e-02 2.50753298192499989374693036836561e-03
            4.37233605166000016684080264894874e-02 9.11023680744600050651627043407643e-01 2.50753366589999979532898954914799e-03
            3.88480061834999973457982491709117e-02 9.67030908282000001285894086322514e-02 3.12341690462499986177813404708559e-03
            9.67032117936000024105425154630211e-02 3.88479942385999976739441308382084e-02 3.12341815312499986034833199255445e-03
            8.73226911312000031450963888346450e-02 8.73226620390999963561284857860301e-02 3.50493272842500010536004317884817e-03
            4.21445202084000034026267655917763e-02 8.48561778910799979769308265531436e-01 3.58340542239999997053967462079527e-03
            8.48561797496099945448122525704093e-01 4.21445420915000021344631875308551e-02 3.58340680312500001539199878664022e-03
            8.47792133386400026395790519018192e-01 1.06743594247199993785280014435557e-01 3.84010356850000020637048159244387e-03
            1.06743588939800002068913897801394e-01 8.47792132814599974643954283237690e-01 3.84010458562499988263705397173453e-03
            1.83396652199100007596399564135936e-01 4.16340521608000030395757562473591e-02 4.61309564034999980419726739455655e-03
            4.16340541167000005584597488450527e-02 1.83396519692999998563109897986578e-01 4.61309657865000000143140823638532e-03
            7.61163225155999989546273809537524e-01 1.94159920285199988354563060966029e-01 4.89584958932500011880017964926992e-03
            1.94159925414399997523062779691827e-01 7.61163215393800052233075348340208e-01 4.89585049985000013428315668306823e-03
            7.57937874717300008597931082476862e-01 4.39826608586000034062379882016103e-02 4.94081878354999965569094300121833e-03
            4.39826512395000007726864055257465e-02 7.57937824230800027969223719992442e-01 4.94081916692499980431341199960116e-03
            3.69760535917999996757821179471648e-02 5.36318607643599953860302775865421e-01 4.97015977547500022171167444184903e-03
            5.36318713434200011747066128009465e-01 3.69760780935000019087333100742399e-02 4.97016214439999996083097144605745e-03
            1.00125694892100006172519499614282e-01 7.91226709354500012949529264005832e-01 5.17954596210000017125762994396609e-03
            7.91226669352400047152684692264302e-01 1.00125755467300001089192562631069e-01 5.17954837232499978871835821792047e-03
            3.79866714176999992202965472642973e-02 4.15741312855800015935159308355651e-01 5.22357678600000006896575399650828e-03
            4.15741402896500023000925239102799e-01 3.79867061534999994609762552499888e-02 5.22358129889999994732718846535136e-03
            6.50710649146299946465887842350639e-01 4.20141226713000026649069695849903e-02 5.37161434712500000104862252214843e-03
            4.20141133438000011213198092718812e-02 6.50710564508400035599322563939495e-01 5.37161465017500002611106069139169e-03
            4.25548444254000013398098190009478e-02 2.92062602348399980023430089204339e-01 5.55545332589999982242368048446224e-03
            2.92062710723999974238296317707864e-01 4.25548546752999995446131720200356e-02 5.55545400507499991665794070172524e-03
            5.38972953818000033443524898757460e-01 4.19303146900500023352975631496520e-01 5.58363263637500027020132975508204e-03
            4.19303182848900002355208016524557e-01 5.38972909360999952177451177703915e-01 5.58363446847499975828066354210932e-03
            6.54947200970000031894358016870683e-01 3.00735263616200010350354432375752e-01 5.61897312365000015793592424984126e-03
            3.00735279091700002940257263617241e-01 6.54947181273100031972944634617306e-01 5.61897451099999979684529449741603e-03
            3.75240077158499996112794860891881e-01 3.45398013075200016608334863121854e-01 5.74253489612500025301988770820572e-03
            3.45398028278600000007259041012730e-01 3.75240069567299994179876421185327e-01 5.74258486094999989063358825092109e-03
            9.94532168760999951695467302670295e-02 1.59830869518699986153720260517730e-01 5.81995940254999982732542207486404e-03
            1.59830935958500008897686939235427e-01 9.94531960132000042440481024641485e-02 5.81996068765000020006628034252572e-03
            1.79732666166700000287548277810856e-01 7.12458543092399976259798677347135e-01 6.73707999117499969909950152668898e-03
            7.12458446194299988007969659520313e-01 1.79732772223999998084664753150719e-01 6.73708267767499958955657035630793e-03
            1.06606567863599996370815858881542e-01 7.00170178417499955436653635842958e-01 7.01096895024999958168088198817713e-03
            7.00170190409600046166360698407516e-01 1.06606585567700001826452194109152e-01 7.01096911517500032129390774571220e-03
            9.93303629801000065313587583659682e-02 6.06564798479599986968935354525456e-01 7.18815675430000029788590865109654e-03
            6.06564805252099992216585633286741e-01 9.93303896768999994293380950693972e-02 7.18815968177499969166133197973068e-03
            1.02322354270399998310558942193893e-01 2.53338157952800013816840873914771e-01 7.47452072657499994035035228989727e-03
            2.53338232493799997424588354988373e-01 1.02322382618900004080941812389938e-01 7.47452306897499960114394923493819e-03
            6.16622671521700027597034932114184e-01 2.76950206057499981326941451698076e-01 7.72510896289999981351170887933222e-03
            2.76950069310899993357821813333430e-01 6.16622790062399950272720161592588e-01 7.72510964889999967447620221605575e-03
            9.04184571873000025865607653940970e-02 4.98152263700100017551619657751871e-01 7.85077542720000014220271822296127e-03
            4.98152276724800002760673578450223e-01 9.04185045148999944064627243278665e-02 7.85077684887500008847904808817475e-03
            9.28231860167999939248417717863049e-02 3.73841851690799975571621871495154e-01 7.97978882559999963541841339065286e-03
            3.73841869922900027489731655805372e-01 9.28232584789999981556007924154983e-02 7.97979170945000045034323221671002e-03
            2.52167884040700007730606557743158e-01 2.52168092569700019733858198378584e-01 8.03574810154999985556401043140795e-03
            5.08750021870799984391453563148389e-01 3.90558054433000012561905123220640e-01 8.25989003470000045237231489636542e-03
            3.90557911673100000005121046342538e-01 5.08750143766099949260706125642173e-01 8.25989079572500041470650700148326e-03
            1.70614146909600000734386071599147e-01 5.26673803955399977283491352864075e-01 8.90422738972499946963345962558378e-03
            5.26673776131200033212564903806197e-01 1.70614225753699999543755438935477e-01 8.90423190135000014455712857852632e-03
            3.48758152762900008880819768819492e-01 2.58805508488600011229863184780697e-01 9.14352974995000060409733322330794e-03
            2.58805359601699980220956831544754e-01 3.48758349170300019359558518772246e-01 9.14353788010000001484911535953870e-03
            1.69661455805300009913594294630457e-01 3.01352218396399973432409069573623e-01 9.14944117475000040506927234673640e-03
            3.01352180687500026401437480672030e-01 1.69661596321900004502580827647762e-01 9.14945134722499990587163409827554e-03
            2.58020240975899983748576005382347e-01 4.58474177447800002038746924881707e-01 9.24864200284999937728969143790891e-03
            4.58474086019799997870904917363077e-01 2.58020381901099993271486710000318e-01 9.24864437647500078898410436067934e-03
            1.84889868349800007285566039172409e-01 1.84889870455099997181847015781386e-01 9.35134059467499986761929164913454e-03
            6.13074033846500054956152325758012e-01 1.92161199406899990682617840320745e-01 9.38875645792499984032541959777518e-03
            1.92161175099400005628069720842177e-01 6.13074039838899964571794498624513e-01 9.38875781324999963972643968190823e-03
            4.18054116059900005009097867514356e-01 1.65061333641599999877414006732579e-01 9.72219233714999991891048125580710e-03
            1.65061264203600011057915253331885e-01 4.18054119924400002794584452203708e-01 9.72219270854999921549488561822727e-03
            5.15920573962500017550780739838956e-01 2.98271900522899979790736324503087e-01 9.81764108870000068440830887084303e-03
            2.98271893575000002218899908257299e-01 5.15920553436200024499669325450668e-01 9.81764506292500012396562425465163e-03
            4.09889460234000024829725816744030e-01 4.09889431779200008598706972406944e-01 9.96917199577499954510351187764172e-03
            ];
        
    case  25
        
        % ALG. DEG.:   25
        % PTS CARD.:  120
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.7e-13
        
        xyw=[
            8.28815950330000043855260827285747e-03 9.84820276886899992341284360009013e-01 3.71835446474999976439884674661585e-04
            4.61842203024100006469865320468671e-01 5.38157796975900049041285910789156e-01 3.72225881549999998432198822229111e-04
            7.10664412390000013730606553963298e-03 8.08423613900000059828698795172386e-03 3.75148609500000011079956507842326e-04
            9.84761314169899981685318834934151e-01 7.00157551339999992962015440411960e-03 3.76480207824999993978382084591772e-04
            5.37444786904900051460742815834237e-01 4.62555213095100004050408415423590e-01 3.82971717875000021349268797266063e-04
            0.00000000000000000000000000000000e+00 4.88767688014000001306413878410240e-01 5.75815862175000042592898363835729e-04
            4.91413192936099973984198641119292e-01 0.00000000000000000000000000000000e+00 5.91226676049999978583560089617777e-04
            7.03459370200000023987119135426838e-03 9.57415805369699945970296539599076e-01 7.18778590274999989138959843870680e-04
            9.56473471422800036734201967192348e-01 3.64655449485000032749937304288324e-02 7.46562218375000034159549855417026e-04
            3.70198792044999988593012574256136e-02 7.09085771659999971466836754530050e-03 7.59604068425000027685312087299963e-04
            1.02412454274699998846287485321227e-01 8.93612559493699998292015607148642e-01 8.02311492200000046272345688436189e-04
            5.92806581150900013810201016895007e-01 4.94517055999999996518390332767012e-03 9.25739960874999976737920182046082e-04
            5.09484223709999982376839255948653e-03 9.96676659189000002658787025211495e-02 9.35179650875000013747484484127881e-04
            8.15620236890000037399683918692972e-03 4.15561148783999967371016737160971e-02 9.61313580574999957360782243398489e-04
            4.24936107567999987733031730385846e-02 9.49486526035200006923275850567734e-01 9.66769466699999961700751072157800e-04
            9.49554350084400033971121501963353e-01 8.17945072920000029736975477590022e-03 9.79813879449999971735230097635849e-04
            8.93278747123899985460582229279680e-01 5.32243262620000004425868667112809e-03 9.89332067200000006904070382063310e-04
            6.93176129270000034726662008210951e-03 9.06540102043300044520890423882520e-01 1.10080629309999996311153491745927e-03
            9.03583903066500049305886932415888e-01 8.94771171076999966009779541309399e-02 1.14767770432499995741926213099759e-03
            9.05665738208999981928570832678815e-02 7.05253420050000034285941907796769e-03 1.17559173587499996016525471986824e-03
            8.39293327870000056833976742609593e-03 6.66317993111099959868681708030635e-01 1.25037109545000002172598918548374e-03
            6.26124568607099973860385944135487e-01 9.21975831529999942381969901816774e-03 1.30969575389999997976075984951194e-03
            6.28015929790000024257601296540088e-03 8.33520746052699990080725456209620e-01 1.36055260229999992055427426862479e-03
            8.27253925736699979864852139144205e-01 1.66513493932999995239541135561012e-01 1.42328122280000000238198154534075e-03
            6.20058753530000009851486098000350e-03 7.42469325522900014391325385076925e-01 1.47768557472500007901605734161876e-03
            1.67690031118500004758686827699421e-01 6.57177435279999999639954921804019e-03 1.49219919217499991451192364877443e-03
            7.19935306956700049596520329941995e-01 6.43545349620000019241761179955574e-03 1.68155475717500003420001863929656e-03
            2.74974009023700005638346510750125e-01 7.18529612071900025682680279714987e-01 1.70769621559999995943912587392788e-03
            7.92575820050000004468593317596969e-03 1.76641137471400005987121062389633e-01 1.73828147780000002650679924443011e-03
            6.99812207519999972260560383574557e-03 2.70476725400400019871938184223836e-01 1.81150676604999996341383017295357e-03
            8.12524877326299965751843501493568e-01 8.22995332100000080710611172207791e-03 1.81820474032500000241552307045367e-03
            7.35369699699999976316711070012389e-03 5.93416787545300028483552523539402e-01 1.82522327117499993942573155436548e-03
            7.28366593541100004749466734210728e-01 2.64881755375199978708167236618465e-01 1.84011666939999998793275892694510e-03
            1.80064230456499996257946349942358e-01 8.11584897668199967135649330884917e-01 1.85299808137500006920139750121734e-03
            2.65810246776200020946134827681817e-01 6.85535254289999958221546449976813e-03 1.87230535840000010538997088360702e-03
            7.08923645200000042410071188214715e-03 3.75763265974400018087919761455851e-01 1.96510168150000000958366186409876e-03
            3.77405430204300007535067607022938e-01 6.14857353375700022013461421011016e-01 1.96554316057500006245661872128494e-03
            3.69649608667999979294549461883435e-02 9.21079230289300054934642503212672e-01 2.01265902664999989102323496581448e-03
            9.20319410980499963415013553458266e-01 4.26025082113999989386066147289966e-02 2.03607151182499989186536204499589e-03
            4.25477806430999971887274568871362e-02 3.72689941794000015318211183057429e-02 2.03697010380000020565716134512968e-03
            6.19127839498299969989147939486429e-01 3.72405571380900024536231285310350e-01 2.31110366530000010260037512921372e-03
            3.76269720917799976867001987557160e-01 8.14364220110000078256540234633576e-03 2.36686587912499990862813525893671e-03
            9.56111149690000022838276549919101e-02 8.77109837260099944877822508715326e-01 2.42830525342500012919710883352309e-03
            3.02473410376999998772085120890551e-02 9.43858903393000048254180001094937e-02 2.49383952877500021313994515992363e-03
            8.73990569175399967249973087746184e-01 3.13198990882999975249312285541237e-02 2.58419509182499992916426378997130e-03
            8.60413373495800049006732024281519e-01 1.04901978204599996558066266061360e-01 2.80658192914999982808810230494601e-03
            3.47307852351999984952257705117518e-02 8.60985646288599970610277978266822e-01 2.85772796862500009715257753839524e-03
            1.04360660834299998067642434307345e-01 3.57152881003999986275765365917323e-02 2.88876418717500015390098333512015e-03
            7.79762282475400025916201229847502e-01 1.87231819926500009376724165122141e-01 3.38939642392499995357524333883248e-03
            1.85865164256000005293945065432126e-02 4.83439767879400017491775543021504e-01 3.39962407249999988903543801654905e-03
            3.24585286617999971281456339511351e-02 7.78347491604200025783200089790625e-01 3.44387034592499993376768507857832e-03
            8.37129390115699978380803258914966e-01 8.04060570155999965180626531946473e-02 3.44902539854999995450057781454234e-03
            8.36602075314999971134000134043163e-02 8.42141481705099947951964622916421e-01 3.46022099759999996609893813115377e-03
            7.84070242501000019830215137517371e-02 8.49927089145000036474186799750896e-02 3.51585049942500010508594776581504e-03
            4.92923864845799974965245837665861e-01 4.89285591470999980945322249681340e-01 3.52478627522500014696005266046086e-03
            1.87063758407299995933925629287842e-01 3.45210858280999982605763420906442e-02 3.55010279977499999121137719271246e-03
            4.89263696702499972790434412672766e-01 1.90774755077000000846876304194666e-02 3.61296061292499994660465389983983e-03
            4.01982618372000027884283213097660e-02 1.69114318710899991238250095193507e-01 3.75614949097500009853045632723934e-03
            7.89425927886500011787518360506510e-01 4.12206731483999999632317212672206e-02 3.82044510304999996100350223571240e-03
            1.68626045642900013188381080908584e-01 7.89486064058499970030879921978340e-01 3.88876810422500007041746172831154e-03
            3.75090191317400012405158804540406e-01 5.89531827201300040464104768034304e-01 4.11427214999999981553946426515722e-03
            3.56362876879999998513959269530460e-02 3.68125621769899991075902789816610e-01 4.13189398932499972472154325942029e-03
            5.88754816480399956191149613005109e-01 3.59968962540999984356382412897801e-02 4.17118886127500025640113534564080e-03
            3.73308082181999997817278824641107e-02 6.79070467353300011303929295536363e-01 4.18523282462499989636794950342846e-03
            2.82076999337399991762964646113687e-01 3.73639992361000025655748402186873e-02 4.21686658402499990605516444475143e-03
            6.81927760331999976983752276282758e-01 2.80333034572499972814085822392371e-01 4.22205575412500002524529563174838e-03
            3.74938324382000018331062562992884e-02 2.63401618001399995261380126976292e-01 4.30217781727500019062215841358920e-03
            6.98407920412699945167389614653075e-01 3.64154673321999966528750292127370e-02 4.36702670660000027980096604096616e-03
            2.65439089407900008499296973241144e-01 6.98071743619300022487550450023264e-01 4.41659749034999972450155070191613e-03
            1.42984844080000006449893135140883e-01 7.61225461845300044316786625131499e-01 4.57419053687499986560194287221748e-03
            7.62355400764699986382311180932447e-01 9.43741220274999931261206143062736e-02 4.58942131147500008425677009427091e-03
            9.34222022749000013952525023341877e-02 1.47979983683200005950553190814389e-01 4.65981423802499985370451085486820e-03
            5.75900447992300001942567178048193e-01 3.82132964169800026787982005771482e-01 4.74452651475000021508154901539456e-03
            3.82242733252500022089748199505266e-01 4.26716362301000004531026377208036e-02 4.79619806444999980526988636597707e-03
            4.11414081674999973503581429667975e-02 5.71808287443200002897469857998658e-01 4.85201105109999992343050578824659e-03
            8.02462538379000028809784339500766e-02 7.70220438204200053000647585577099e-01 4.86800180482500031986514699156032e-03
            7.62522981940999988736962222901639e-01 1.55942057736200012652005852942239e-01 5.02137701237499994161295191474892e-03
            1.52494144513100005067229858468636e-01 8.42965421322000030279752991191344e-02 5.04184773330000005758799730415376e-03
            6.22159195833000000508405946675339e-02 4.53818131887300024729370306886267e-01 5.54355406902500014165724451231654e-03
            1.10953903607600004588285003137571e-01 4.58601407117100012822419330404955e-01 5.74256101270000009900718040967149e-03
            4.57562721205700018156647956857341e-01 4.79531356020999999234533106573508e-01 5.83662793497499981371090527204615e-03
            4.32286513637400005016786508349469e-01 1.23059123747200002685353581455274e-01 5.87207838345000024404818361745129e-03
            5.86500285024100009856340420810739e-01 8.34119779793000004630698640539777e-02 6.01705247544999984737890130759297e-03
            8.69359250817999945626368685225316e-02 6.75567701335100023740665164950769e-01 6.02276982382500027102256723310347e-03
            9.29594906936000026087540959451871e-02 2.32650089272700005826877145409526e-01 6.14192623702500013066485351487245e-03
            6.66193214145400003367569752299460e-01 2.44829400740600000574431760469452e-01 6.16340789297499958704351641358699e-03
            4.78030636222699978343797511115554e-01 6.61749044834999966546718042081920e-02 6.16891325129999973375660715646518e-03
            4.37221529457699986664920288603753e-01 4.44214558524400005001808722226997e-01 6.24261506775000002411601940366381e-03
            6.77922450466900028409611422830494e-01 9.29096534576999977250721940436051e-02 6.25066360205000018379628556886018e-03
            2.42343125565999989445842288660060e-01 8.89793655129000049530674232300953e-02 6.26227173564999992777524440157322e-03
            2.28892542030499995808767721428012e-01 6.78005308167199949309633666416630e-01 6.27340625312499972771052014763882e-03
            3.31506504995900008747611309445347e-01 5.84738155974099949041544732608600e-01 6.28705190565000040375798207037406e-03
            3.42420052660700025004558710861602e-01 5.13924572273599999938653581921244e-01 6.37525726117499965533985673005191e-03
            8.62630046475000022976331592872157e-02 3.34097624923400005059193063061684e-01 6.41361279947500004944282991914406e-03
            5.11318894663500000952183199842693e-01 1.38015472055399990303214963205392e-01 6.44936876575000032907336589005354e-03
            1.53897784100099993542798415546713e-01 6.78806261956199974427761389961233e-01 6.75019384982500011971318087944383e-03
            6.77995134847199976135811994026881e-01 1.66335892526900008814294551484636e-01 6.86078842109999996468072680499972e-03
            1.66460046941100003881075508616050e-01 1.58221450484899994748388962761965e-01 6.92681003720000028345271658736237e-03
            9.50910318887999983505920909010456e-02 5.66659033254300004145420643908437e-01 6.95711038409999992654375589040683e-03
            3.43604813671199993407867623318452e-01 9.78960873457000030350982910931634e-02 7.18018452762500015984503676236272e-03
            5.56041702536600013218048843555152e-01 3.46891782094700007466059332728037e-01 7.22067087390000032082193470728271e-03
            1.45240402951300001666012917667103e-01 3.59953449105199996260751049703686e-01 7.33256824397500017037598851743496e-03
            1.61968515623799991232090178527869e-01 5.81013137332999973772018620365998e-01 7.97257198892500039411057599636479e-03
            5.80016484426200018553743120719446e-01 2.56067464067199990207512882989249e-01 7.97709150714999946918393192163421e-03
            2.45020122328799994093628811242525e-01 5.88146955210199995356390445522266e-01 8.02347400822499973171986198394734e-03
            2.55762189179400023508748063250096e-01 1.65224406504699994346907487852150e-01 8.04046521949999988543300588617058e-03
            2.20523998551099997822433351757354e-01 3.49650746610600027253212829236872e-01 8.06060318835000004511481819235996e-03
            4.94018311128499976181416286635795e-01 2.54944844845300000457655187346973e-01 8.17681116052500038537953486184051e-03
            2.53157068979800004004943048130372e-01 2.54336911501700013715776549361181e-01 8.24865791737500013425243849951585e-03
            5.84689111635700031754936389916111e-01 1.66660391647900002487503456904960e-01 8.29570240062500047217852738867805e-03
            1.66033360227799992037134302336199e-01 2.52324019170500013498070757123060e-01 8.37142906627500078819981155220376e-03
            2.50542629246099990414364810931147e-01 4.95900762752799983967832986309077e-01 8.38671181980000027267330153790681e-03
            3.51933680218199984590654594285297e-01 1.80538036779999994108081295962620e-01 8.42622607469999986007369585649940e-03
            3.50266883541899998188995368764154e-01 4.35858232988100002813780520227738e-01 8.50903656917499987000219618948904e-03
            4.40089248551200018511053713154979e-01 2.12057610494099990638972030865261e-01 8.56163088307499917095633179542347e-03
            4.68085547154599990182788360471022e-01 3.55268157077400004162370805715909e-01 8.63822043127500036052701659627928e-03
            1.77023776394699988712133631452161e-01 4.67035292226600007303716211026767e-01 8.91957189257499942491858035964469e-03
            3.90092077950099980210296735094744e-01 3.32315281929999983834278509675642e-01 9.11640562540000018443286933234049e-03
            2.80584777411999997109859350530314e-01 3.89804117667999994178984479731298e-01 9.12931771764999983043153974904271e-03
            3.36152334744000025334997872050735e-01 2.77850004435599984287108554781298e-01 9.29812027544999961103933117101406e-03
            ];
        
        
    otherwise
        xyw=[];
end




pointset_stats=[
    2    3 1.73e-13    0    0
    4    6 1.87e-13    0    0
    5   10 1.77e-13    0    0
    7   15 4.59e-13    0    0
    9   21 3.33e-13    0    0
    11   28 5.89e-13    0    0
    13   36 3.94e-13    0    0
    14   45 4.53e-13    0    0
    16   55 7.82e-13    0    0
    18   66 5.01e-13    0    0
    20   78 7.68e-13    0    0
    21   91 8.94e-13    0    0
    23  105 7.02e-13    0    0
    25  120 5.66e-13    0    0
    ];




