
function [xyw_bar,pointset_stats]=set_taylor_wingate_bos_siam_C_barycentric(deg)

%--------------------------------------------------------------------
% CUBATURE RULE ON THE SIMPLEX [0 0; 0 1; 1 0] WITH ADE EQUAL TO deg.
% THE CUBATURE RULE IS DESCRIBED IN BARYCENTRICAL FORM.
%--------------------------------------------------------------------
% INPUT:
% deg: algebraic degree of precision (ADE) of the rule.
%--------------------------------------------------------------------
% OUTPUT:
% xyw_bar: MATRIX OF NODES xyw(:,1:3) IN BARYCENTRICAL COOR. AND WEIGHTS xyw(:,4).
%      REMARK: The sum of the weights makes 1.
%
% pointset_stats:
%    statistics of the pointset stored in a matrix. Each rows contains
%    degree, cardinality of the full pointset, moment error w.r.t.
%    orthn. Dubiner basis, number of points with negative weights,
%    number of points outside the domain.
%--------------------------------------------------------------------
% SOURCE:
% M.A. Taylor, B.A. Wingate, L.P. Bos,
% "A cardinal algorithm for computing multivariate quadrature points",
% SIAM J. Numer. Anal.,
% Vol. 45, no.1, (2007),  pp.193-205.
%--------------------------------------------------------------------
% | DEG | CARD | W- | PT.O|   M.E.   |Q.|
%--------------------------------------------------------------------
% |  13 |  36 |   0 |   0 | 5.69e-16 |PI|
% |  16 |  55 |   0 |   0 | 7.82e-13 |PI|
% |  18 |  66 |   0 |   0 | 6.28e-16 |PI|
% |  20 |  78 |   0 |   0 | 6.59e-16 |PI|
%--------------------------------------------------------------------
% LEGEND:
%--------------------------------------------------------------------
% DEG  : algebraic degree of precision
% CARD : cardinality of the pointset
% W-   : negative weights
% PT.O : points outside the simplex [0 0; 1 0; 0 1].
% M.E. : maximum moment error in inf norm.
% Q.: quality of the rule.
%        PI: pos. weights, points in simplex
%        NI: some neg. weights, points in simplex
%        P0: pos. weights, some points out of the simplex
%        N0: some neg. weights, some points out of the simplex
%--------------------------------------------------------------------


switch deg
    
    case  13
        
        % ALG. DEG.:   13
        % PTS CARD.:   36
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.7e-16
        
        xyw_bar=[
            2.42935351590267065158279535808106e-02 9.49305929384640645452009266591631e-01 2.64005354563326966044201071781572e-02 8.31204993785483389756585381746845e-03
            2.65193427721596430846062730779522e-02 2.42695130640411330102956810605974e-02 9.49211144163799258599567565397592e-01 8.34058498887745372596391746355948e-03
            9.49212602355106627172176558815408e-01 2.65067966437244444122267594821096e-02 2.42806010011689284155966817024819e-02 8.34152845332809290967990278886646e-03
            3.37757637490055826964829854830441e-03 4.76731641236311565190675310077495e-01 5.19890782388787942025487609498668e-01 8.78404350412634662981226085776143e-03
            4.75767229810120817479202059985255e-01 5.19892182910199451839616813231260e-01 4.34058727967978619233235804131255e-03 9.22373309223947085433437109713850e-03
            5.19078319347068384281840280891629e-01 5.59127062021519405377034672710579e-03 4.75330410032716421664389372381265e-01 9.89712050939172034713653403059652e-03
            8.61683974532058805095857678679749e-01 1.33996048618149864173343388529247e-02 1.24916420606126205017361030513712e-01 1.01770197927333848342268396436339e-02
            1.24920975992555069455924865451379e-01 8.61305432133411819073387505341088e-01 1.37735918740331531040510526509024e-02 1.03426431970092824741414005984552e-02
            1.38565453861037562316438354059756e-02 1.24773371735846960728366639159503e-01 8.61370082878049281305266049457714e-01 1.04135683043232209199491222761935e-02
            2.11887064221680555820004343559049e-02 8.43843835122266905379717627511127e-01 1.34967458455565014752153274457669e-01 1.58909889139352972220464721431199e-02
            8.43229678721886810599528416787507e-01 1.35456364583034560844509996968554e-01 2.13139566950786285559615862439387e-02 1.60236017620355164492451649493887e-02
            1.35423179786502895227684462042816e-01 2.13482820656213760635022680389739e-02 8.43228538147875794628305357036879e-01 1.60303840572841652112323629353341e-02
            3.08885351067946345438031130470335e-01 2.21919663013588897015893053321633e-02 6.68922682630694764860379564197501e-01 2.15382979591376422234816345735453e-02
            6.68505759516906161721294665767346e-01 3.08901287938941060406961014450644e-01 2.25929525441527778717443197820103e-02 2.19236707669562061462364965791494e-02
            2.26545012557139990860832767793909e-02 6.69170994332103363788633032527287e-01 3.08174504412182637125283690693323e-01 2.19604836366373568079524147833581e-02
            2.80851540877201766299720020469977e-01 6.92471815510625865464078287914163e-01 2.66766436121723682362016916158609e-02 2.39975961845282498297038387136126e-02
            6.92244674905063805958604916668264e-01 2.68723345025942451924638731952655e-02 2.80882990592341952318378162090085e-01 2.41903130366623898839240069946754e-02
            2.68617447119446836611977857955935e-02 2.81009397322192211454705557116540e-01 6.92128857965863097945202753180638e-01 2.42433711687330931405792711075264e-02
            1.14177848547014307301061819543975e-01 7.97358141358575300827737919462379e-01 8.84640100944104057489880688081030e-02 2.78482244012102746200110203744771e-02
            7.97480792206136435140706453239545e-01 8.79806508790785396501021864423819e-02 1.14538556914785025209191360318073e-01 2.80513182178032081937057995446594e-02
            8.92807293894229969666298529773485e-02 1.14502056112754047045854122188757e-01 7.96217214497823011498667256091721e-01 2.82595061846439739305925087364812e-02
            1.05248789245499130284855482386774e-01 6.68690411992204802871242463879753e-01 2.26060798762296011332750822475646e-01 3.44644945334760116617545122608135e-02
            6.66302228074011737568582702806452e-01 2.27505163183173875207643277462921e-01 1.06192608742814387223774019730627e-01 3.58606668044348930735054636897985e-02
            2.30780373754708834077575829724083e-01 1.05457256122134593323558249267080e-01 6.63762370123156530965502497565467e-01 3.63726960488191183862305422280770e-02
            1.70505915754032344455737302268972e-01 5.17406439865800038546694850083441e-01 3.12087644380167672508719078905415e-01 3.94403668368501700292050315965753e-02
            5.08659397304299987219167178409407e-01 3.17052385520937729523893722216599e-01 1.74288217174762283256939099373994e-01 4.05057172756122657597543934571149e-02
            3.14182386228100019032183354283916e-01 1.81070636165945680895461578074901e-01 5.04746977605954327827930683270097e-01 4.12862649527448860586886780765781e-02
            4.61746081786398032686946635294589e-01 4.67859453980425443742063862373470e-01 7.03944642331764680598382710741134e-02 4.21022283665200211522083861837018e-02
            6.93087496081041115614240766262810e-02 4.62285604208458600350439837711747e-01 4.68405646183437329721499509105342e-01 4.21792766652515915826171521985088e-02
            4.65195525926812680772570729459403e-01 7.24357805669118837155195933519281e-02 4.62368693506275407756334061559755e-01 4.25984934243976501311301774421736e-02
            2.57862585789255804868957966391463e-01 6.13139503917723982517884451226564e-01 1.28997910293020212613157582381973e-01 4.51422664026207703935611448287091e-02
            6.11262776677922614609883567027282e-01 1.30036083460934620559257268723741e-01 2.58701139861142737075283548620064e-01 4.57141571742253824406354567599919e-02
            1.30518213593359244661940010701073e-01 2.58171382888366274244162923423573e-01 6.11310403518274481093897065875353e-01 4.58139532704635935034254146103194e-02
            4.28143799182839956962709493382135e-01 2.36200596981703037080535523273284e-01 3.35655603835457005956754983344581e-01 5.12786687447955233376539752043755e-02
            3.35699578373030105282737167726737e-01 4.31102630858835489391367445932701e-01 2.33197790768134405325895386340562e-01 5.16579830706738740708772184007103e-02
            2.30542429883610316965203423933417e-01 3.45601394937618766611819864920108e-01 4.23856175178770944178552326775389e-01 5.17927183596631479645111539866775e-02
            ];
        
    case  16
        
        % ALG. DEG.:   16
        % PTS CARD.:   55
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 7.8e-13
        
        xyw_bar=[
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 3.10129992550000005771537736620758e-04
            0.00000000000000000000000000000000e+00 1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 3.15758735599999988078595558604889e-04
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 1.00000000000000000000000000000000e+00 3.54330077950000001639485613935676e-04
            9.39886358357699958077091650920920e-01 4.98487446340000004557646562375339e-03 5.51287671789000427446936214437301e-02 2.75818580839999995604761018341833e-03
            5.43806683058000034369783293186629e-02 9.38640561861699973533745833265129e-01 6.97876983250000915148802960175090e-03 3.13462038279999988601987048753017e-03
            9.39400491640000022408596436207517e-03 5.26424462697000009048231561337161e-02 9.37963548813899983258579595712945e-01 3.92657044129999990150770017294235e-03
            1.64345086361999993818105991749690e-02 9.46903551735099990871447062090738e-01 3.66619396287000132161892906879075e-02 4.72757419320000028123418545078493e-03
            9.46948726986200028932216810062528e-01 3.63373677167000003196406510141969e-02 1.67139052970999707481425389232754e-02 4.89122556354999998146082518246658e-03
            4.26604005767999977516424792156613e-02 1.51224541798999998626351626285214e-02 9.42217145243300024937127545854310e-01 4.99308217444999983408715493737873e-03
            1.22269495439000005360119516240047e-02 8.69377351066400017032265168381855e-01 1.18395699389699959880317692295648e-01 6.87769094080000007590536270640769e-03
            8.67369652104699961903122584772063e-01 1.20491728577399995203656146713911e-01 1.21386193179000428932212685140257e-02 7.04895890199999999858881949421630e-03
            8.45674402138899994341159072064329e-01 1.57763967870000014481757233397730e-02 1.38549201074100014619006060456741e-01 7.48234321684999956897366146790773e-03
            1.39575963210299996974939062965859e-01 8.44812087037499992447919794358313e-01 1.56119497522000383327167583047412e-02 7.80487518060000041303370110767901e-03
            1.31782174323099993129559948101814e-01 1.35009605583999992006738466443494e-02 8.54716865118499979914190589624923e-01 7.88418466740000052628278837119069e-03
            1.57955126300000012196100840355939e-02 1.45527493853600004003467915936199e-01 8.38676993516400015593603711749893e-01 8.78972731915000048130703902415917e-03
            7.36546288443600039030911830195691e-01 1.55697540907999993492483170598462e-02 2.47883957465599963354563328721269e-01 1.02056920135000003119207789836764e-02
            1.39688430330000001866608272393933e-02 7.37983689445000035078692235401832e-01 2.48047467521999909223495706100948e-01 1.04781439308000001026099923251422e-02
            2.54789518603899978188565000891685e-01 7.29761568977099983612788491882384e-01 1.54489124190000381986465072259307e-02 1.05356706499000005866628626449710e-02
            7.31638652255499954968342990468955e-01 2.54307668331499980851617692678701e-01 1.40536794130000641800393168523442e-02 1.08823380101000007774691979989257e-02
            1.57253728950999989433245929149052e-02 2.69623979579100003789449147006962e-01 7.14650647525799986858885404217290e-01 1.11144204349499994832761728957848e-02
            2.66230284364699976151769078569487e-01 1.44783956308000005819280886498746e-02 7.19291320004500023266302832780639e-01 1.12093346841000002234167709502799e-02
            8.67350406521400052461956420302158e-01 5.91679410399999980429974755224976e-02 7.34816524385999425561522002681158e-02 1.15061308496499994924455734235380e-02
            7.41493666957000058248539176020131e-02 8.63478257506099966533952283498365e-01 6.23723757981999860078303754562512e-02 1.18406951249999999736806088890262e-02
            1.59285948359999991741364766539846e-02 4.19123895523800027440586291049840e-01 5.64947509640200018488087607693160e-01 1.28732321684000006190018439156120e-02
            1.56061028068000002405613102496318e-02 5.80922292114600002932434108515736e-01 4.03471605078600026317303672840353e-01 1.28978400803999998514015601358551e-02
            5.91009481748399956302364444127306e-01 1.59251452650999991700331293031923e-02 3.93065372986500016772026810940588e-01 1.29036163804999992610333947595791e-02
            4.03477149688900027513227541930974e-01 5.80670036810399969873230929806596e-01 1.58528135007000026135415282624308e-02 1.30171616029500000072394172434542e-02
            5.69474562852600008078240989561891e-01 4.14949514630200022580197583010886e-01 1.55759225171999693415614274272230e-02 1.32884070804499994367109394488580e-02
            6.78493700649999975649251382492366e-02 7.61218678590999953126328136931988e-02 8.56028762075899951611290816799738e-01 1.32892380915500003119023020303757e-02
            4.26596859027200014935488070477732e-01 1.57509692312000001979122032480518e-02 5.57652171741599977927705822366988e-01 1.33766164619000000463033828168591e-02
            6.70982507890000023920151761558373e-02 7.74189831242099946173595981235849e-01 1.58711917968900051434388842608314e-01 1.87893903320500016418392874584242e-02
            7.52831023147999989220124916755594e-01 8.19119495638999939313151799069601e-02 1.65257027288100016848559903337446e-01 1.91532947097499996191860560657005e-02
            7.75372778355699954211388558178442e-01 1.57712845729199996336333811086661e-01 6.69143759151000494522776307348977e-02 1.92424847512500003798852077352421e-02
            1.68907315778699995689748902805150e-01 7.50394309974199980040054924756987e-01 8.06983742471000242701961724378634e-02 1.94809912925999993704717638820512e-02
            1.68733583291899996714491294369509e-01 7.08311507268000045689504418078286e-02 7.60435265981299957083194840379292e-01 1.97302055773499983304386518057072e-02
            8.21244708436000003803911795330350e-02 1.76299662677100010821362729984685e-01 7.41575866479299961042670474853367e-01 2.06182389048999992198218933481257e-02
            6.28870536334499963260213917237706e-01 8.07744953317000025094785087276250e-02 2.90354968333800034230307574034669e-01 2.56436219241500014398571494211865e-02
            8.11413015265999987279954552832351e-02 3.05437358977599993092866270671948e-01 6.13421339495799911034623619343620e-01 2.58202820967500003279671005884666e-02
            2.96911206508000014014925227456843e-01 6.22748598887099991117111130733974e-01 8.03401946049000503791148730670102e-02 2.59115021134499988464394704124061e-02
            7.67542314170999995237565372008248e-02 6.24724714954600024441333516733721e-01 2.98521053628299948279334330436541e-01 2.64263994090499984568332081380504e-02
            6.22302233384499969837122534954688e-01 3.01148582116600005420536945166532e-01 7.65491844989000247423405198787805e-02 2.69252786513499986487119031153270e-02
            3.10378628805099976428749641854665e-01 7.79098365078999960386951784130360e-02 6.11711534687000013654767371917842e-01 2.70947664659500001005376645935030e-02
            8.19218215187000053933985554976971e-02 4.60363303835099990646284595641191e-01 4.57714874646200031715892464490025e-01 2.92368573222000008304810592107970e-02
            4.71702266501299982692785306426231e-01 8.21554006796999941597547945093538e-02 4.46142332819000009269672091249959e-01 2.96431584181500000496622959644810e-02
            4.54660341525000011309032288409071e-01 4.63756503388999985482143983972492e-01 8.15831550860000032088237276184373e-02 2.97179138374500002861555714162023e-02
            1.70109133923700001078671562027012e-01 6.42227780818800053630468482879223e-01 1.87663085257499973046435570722679e-01 3.15900127931499977163021242176910e-02
            6.40600432948699949342596937640337e-01 1.89829353725600002977813574034371e-01 1.69570213325700047679589488325291e-01 3.16463422576499994742782462253672e-02
            1.91226758371700000793680374044925e-01 1.73995568534300010243853762403887e-01 6.34777673094000016718041479180101e-01 3.20353680885999969607524917591945e-02
            1.88531576706999992598667859056150e-01 4.79891407040600015410092282763799e-01 3.31577016252400047502391089437879e-01 4.06020297959000034393461930903868e-02
            4.77292995769100003577989355108002e-01 3.34835659811899999382944770331960e-01 1.87871344418999997039065874560038e-01 4.07218756764999995079179484491760e-02
            3.12697462175999973954532151765306e-01 4.95797219725900017994746349359048e-01 1.91505318098100063561872730133473e-01 4.07339600620499966976417738351302e-02
            4.96122594594600008210960595533834e-01 1.92755366890399987278570392845722e-01 3.11122038515000087777195858507184e-01 4.07525274041999965857208110264764e-02
            1.92880531286699991033728451839124e-01 3.16101580726100006568657363459351e-01 4.91017887987199974642038569072611e-01 4.07582332469499999949569257751136e-02
            3.36004145381600005659805674440577e-01 1.89489280128999987296012363913178e-01 4.74506574489399923777455114759505e-01 4.08465529811500027701498538590386e-02
            3.33728055084800023788460521245725e-01 3.34357102181099985482859437979641e-01 3.31914842734099935217528809516807e-01 4.61609167265499970023157061405072e-02
            ];
        
    case  18
        
        % ALG. DEG.:   18
        % PTS CARD.:   66
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.3e-16
        
        xyw_bar=[
            1.16731059668422516850672110422238e-02 9.81256595128901731861503776599420e-01 7.07029890425603380066377212642692e-03 1.25828784932243190573808444554516e-03
            9.81003085838795807482881627947791e-01 7.14625048632000492432325700065121e-03 1.18506636748841875927951150515582e-02 1.26367260036098442294028032506503e-03
            1.06966317091656843568658885601508e-02 1.15153933375963680313391890308594e-02 9.77787974953237992714605297805974e-01 1.66346476665858397127328860420903e-03
            9.38247698355050507856844888010528e-01 4.95570591340609920005810806742375e-02 1.21952425108885001425740313152346e-02 4.07517460627058927119525932880606e-03
            1.26627518417216332335950568221961e-02 9.37012362061508063959536229958758e-01 5.03248860967703270929973768943455e-02 4.30677628708067087442978859712639e-03
            5.98109409983654805031960677297320e-02 1.21364578921857504262282745344237e-02 9.28052601109448760396958277851809e-01 4.38933730896490584272306989532808e-03
            1.37363297926713905788842495780955e-02 6.12783625596971526228173843264813e-02 9.24985307647631538330301737005357e-01 4.85497927808363049367867958494571e-03
            9.22952795940536807606235925049987e-01 1.41128270602437471542556224335385e-02 6.29343769992194435047849765396677e-02 5.12331059574435213055387805525243e-03
            6.33107354992724497000367023247236e-02 9.22019729172741220324382993567269e-01 1.46695353279863160977924962935504e-02 5.41988441703700023210910075022184e-03
            1.17265100334620030186894013013443e-02 1.50052047522907017196303058881313e-01 8.38221442443630948559984972234815e-01 6.46926950879339515004806671072402e-03
            1.55472058732349538434647229223629e-01 8.32514712158920922036031697643921e-01 1.20132291087295950404723043902777e-02 6.81699117914835121118111160853914e-03
            8.34329388898202961577510450297268e-01 1.25228158758818562679149266614331e-02 1.53147795225915178685127671087685e-01 6.92386640733264665387158842690951e-03
            8.50163803195680189972449625201989e-01 1.37199750873573778076064400011091e-01 1.26364459307460319514859747869195e-02 6.97107700524266134245454651363616e-03
            1.28816350521985759775001056937072e-02 8.47762706347928718386697255482432e-01 1.39355658599872755942783442151267e-01 7.20606999838004861486107088808239e-03
            1.51080160895852744218359475780744e-01 1.36526924039371434088652179639212e-02 8.35267146700210072474135358788772e-01 7.68517277670056810773280275839170e-03
            1.01917879216608352199724407682879e-02 5.77043861834499960394850859302096e-01 4.12764350243839173160154132347088e-01 8.12449011263039616581593804767181e-03
            2.81337239930328053549857258985867e-01 7.06685375962319506371045463311020e-01 1.19773841073524955902485089609399e-02 8.48591521400748720338480524105762e-03
            7.12437462850067815267607329587918e-01 1.24569780989904569518111898673851e-02 2.75105559050941750331986668243189e-01 8.50442662106709601488319094642065e-03
            2.76302525086301542511790785283665e-01 1.21741311385635160580154234821748e-02 7.11523343775134931021852935373317e-01 8.54767603373145223866202968565631e-03
            1.09658368560643058875481159475385e-02 4.19430671246641517679876187685295e-01 5.69603491897294089696401897526812e-01 8.69444272795685639543261657991025e-03
            4.28911051788389230399189955278416e-01 5.59961606746893991726210515480489e-01 1.11273414647168333857507604989223e-02 8.72719812193595463611917040225308e-03
            4.21542055511432756098599838878727e-01 1.16475994784656037422942986836460e-02 5.66810345010101634954935434507206e-01 8.92033786433164659068761892513066e-03
            5.71125859044400052688672531076008e-01 1.18218313988502580352824011811208e-02 4.17052309556749711827450255441363e-01 8.92234319396820706182804627815131e-03
            5.82686827051092648410701713146409e-01 4.05788958117720421903129590646131e-01 1.15242148311869296861686962074600e-02 8.95231687761671550418451914765683e-03
            1.30567806713243281940828310894176e-02 2.72502375086800530112896012724377e-01 7.14440844241875150366638536070241e-01 9.06298781003534066613713804372310e-03
            1.30760400963916331024128325566380e-02 7.22471252323362289615715781110339e-01 2.64452707580246082486041814263444e-01 9.23924194410074249572684124132138e-03
            7.26343706240680120345132309012115e-01 2.60298401925062650175135559038608e-01 1.33578918342572294797321319492767e-02 9.28967821855570217992603687662267e-03
            6.87230068637360441696770863018173e-02 6.31417277209675720239800966737675e-02 8.68135265415296397684130624838872e-01 1.01608575888281586763195463163356e-02
            8.65230210152928336242439399939030e-01 7.20611837337740562814403233460325e-02 6.27086061132976074761202767149371e-02 1.06885830904604357816101156686273e-02
            6.48599071037404134587589510374528e-02 8.59043354390996882763431585772196e-01 7.60967385052626621444460397469811e-02 1.15958427049141516579355837279763e-02
            1.48349494336221382484453101824329e-01 7.88878835223944530063988622714533e-01 6.27716704398341152071338910900522e-02 1.37213355429586479627968031991259e-02
            6.24359898395993248576552048234589e-02 1.49393549935431552144748934551899e-01 7.88170460224969171569853187975241e-01 1.45150961170182283416218993465918e-02
            7.87136901173496439376719990832498e-01 6.56382042756524786142335869953968e-02 1.47224894550851082009046422172105e-01 1.47261369252703593762943157230438e-02
            5.19104921609671449145295696325775e-02 5.25563569560499987964874435419915e-01 4.22525938278532908753959418390878e-01 1.49718125814584396976458791073128e-02
            1.54312992744386645593834828105173e-01 7.16383926916975000143139595820685e-02 7.74048614563915826636275596683845e-01 1.53513474059392782816724221106597e-02
            2.61784274560300012080205078746076e-01 6.21479485288090535699545569059410e-02 6.76067776910890927410946460440755e-01 1.62631682931342752551628905166581e-02
            7.66725787281275161788585137401242e-01 1.65821155483138527042541454648017e-01 6.74530572355863111688734079507412e-02 1.63942104253044507178316280260333e-02
            2.58210367662758677465006940110470e-01 6.80011976613870916530402155331103e-01 6.17776557233703504934396733005997e-02 1.65617337596012451084170891135727e-02
            6.79065925147462196020953228980943e-02 7.57151543778184943356279745785287e-01 1.74941863707068878674988354759989e-01 1.73083763437293924880577833391726e-02
            5.29357827480447351575776337995194e-01 4.12150384110698153605767402041238e-01 5.84917884088544948184562599635683e-02 1.73540686988042300309409426972707e-02
            6.66036150484166228347149285582418e-02 2.61251308788644442948623236588901e-01 6.72145076162938948094449642667314e-01 1.73686024701916186252503848663764e-02
            5.85675461899543767230191804173955e-02 3.90223611453476837418463674111990e-01 5.51208842356568751164047625934472e-01 1.74264381227117980122098828132948e-02
            6.44535360410797703023533244959253e-02 6.37362655976104086619216104736552e-01 2.98183807982816184711793994210893e-01 1.74300780592989701389683432353195e-02
            6.74813842915122918952874897513539e-01 6.37583342061289193303252886835253e-02 2.61427822878748161716799813802936e-01 1.77735784987432775261506634478792e-02
            3.91460231036899974910170385555830e-01 5.50323809056281976204161310306517e-01 5.82159599068180488856683041376527e-02 1.80091498191344533130742178173023e-02
            6.48770149230724513245149864815176e-01 2.83672836026289454380844290426467e-01 6.75570147429860323740058447583579e-02 1.81463142921354618486429188806142e-02
            3.94649822040804953982018332681037e-01 6.05175522553690892202382656250848e-02 5.44832625703825845775440939178225e-01 1.90948851041584030663322835152940e-02
            5.39013715193328524222238229413051e-01 6.11990176936390989359537684322277e-02 3.99787267113032362964020194340264e-01 1.96126400058931373016068988590632e-02
            1.62789508278483069370068392345274e-01 6.86132214103465032373208032367984e-01 1.51078277618051926012299190915655e-01 2.41355062943759066107496380482189e-02
            6.81243632264075360538413406175096e-01 1.56796834589900763168301978112140e-01 1.61959533146023876293284615712764e-01 2.44956060783115964207823367360106e-02
            1.54283287802026936041954741085647e-01 1.66751262401978794791546079068212e-01 6.78965449795994269166499179846141e-01 2.48610416936092572248728771455717e-02
            2.52272775044500019436810589468223e-01 2.50480393339460316592948174729827e-01 4.97246831616039719481392467059777e-01 2.53532868492960250517143094839412e-02
            2.54798153240699987165385209664237e-01 4.99409064904299981879631786796381e-01 2.45792781855000030954983003539382e-01 2.54885997021513828475480778479323e-02
            1.48558054919430054807705232633452e-01 5.75602309608733819779047280462692e-01 2.75839635471836097657671871274943e-01 2.60680031833592160128709736000019e-02
            2.93023960643626513178361392419902e-01 5.65689735416167960302402661909582e-01 1.41286303940205582030387176928343e-01 2.61730437462385605307257918639152e-02
            2.80899127230993417292381764127640e-01 1.43792157424766281481609553338785e-01 5.75308715344240328981584298162488e-01 2.62220341775821451368866377151789e-02
            4.82098959297124929346267663277104e-01 2.51855753586498309903873860093881e-01 2.66045287116376705238707245371188e-01 2.63729822411242591229108001016357e-02
            5.64187824544361671286196724395268e-01 1.46296674315256031162491012764804e-01 2.89515501140382269795736647211015e-01 2.64724531863810459697550214741568e-02
            1.30769964434412994114254047417489e-01 4.48957758611656820857405136848683e-01 4.20272276953930157272765200104914e-01 2.71197797250435496119180811547267e-02
            1.47969222194773181167093412113900e-01 3.00117438682896475743433484240086e-01 5.51913339122330315333897488017101e-01 2.71735101709683450621479749997889e-02
            5.63868422294612758882692560291616e-01 2.81377208929746214227662903795135e-01 1.54754368775641026889644535913249e-01 2.73550274319399773503835859855826e-02
            4.36115742879062906478537797738682e-01 4.25205344642040350411349436399178e-01 1.38678912478896687598961534604314e-01 2.78644172956311712840982153238656e-02
            3.60326393528605848359802621416748e-01 2.59919000488856577479879206293845e-01 3.79754605982537574160318172289408e-01 2.88867132116526689755708900975151e-02
            4.22418833467427368155711064900970e-01 1.45323844330263962643101649518940e-01 4.32257322202308669201187285580090e-01 2.92696890811342674831863064355275e-02
            3.71900183305260034760664211717085e-01 3.78012270356697988038519042675034e-01 2.50087546338042032711967976865708e-01 3.04519625339803971797447701419514e-02
            2.41364500692874683895894349916489e-01 3.84756328493939425694492229013122e-01 3.73879170813185945920764652328216e-01 3.18636982224730774815313338876877e-02
            ];
        
    case  20
        
        % ALG. DEG.:   20
        % PTS CARD.:   78
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.6e-16
        
        xyw_bar=[
            8.94113371120544564440990598086501e-03 8.69832937019904967046457500146062e-03 9.82360536918595483868443807295989e-01 1.08722726992999639550330215342910e-03
            9.79262262980672537615589590132004e-01 1.02644133743667025776558077154732e-02 1.04733236449607598067546021525231e-02 1.44935676325573504184818229845177e-03
            1.05475382111858764777601038531429e-02 9.78551420251513315307079210469965e-01 1.09010415373007596429033583262935e-02 1.54230146682676587388538536060878e-03
            2.37770619471535044411880654990910e-03 6.36551098603542475640182374263532e-02 9.33967183944930390282479493180290e-01 1.72008165518023302398287199821425e-03
            6.30425115794741308672399782153661e-02 4.15063475086174891309775958347927e-03 9.32806853669664093331448384560645e-01 2.09492360058795203808790219568436e-03
            9.30842249673007549048975306504872e-01 4.80534822625918074667783486120243e-03 6.43524021007332719390703346107330e-02 2.23690257489723553357974239474970e-03
            6.29076555490187538932644883971079e-02 9.31679006948127330645093024941161e-01 5.41333750285388770606687103281729e-03 2.35272104071028442806579050738947e-03
            9.31596224638061132239386097353417e-01 6.26264881801440048691986817175348e-02 5.77728718179486289141522092904779e-03 2.44339678751489142866093473571709e-03
            6.19516894144997096183224982723914e-03 9.29358705856405231671146793814842e-01 6.44461252021447661419983887753915e-02 2.59638216844565233518138391843877e-03
            2.87125819236767679254462848348339e-02 3.10202122997133844517492207160103e-02 9.40267205776609826806122782727471e-01 3.70365294904030162639885226383285e-03
            9.29384447830515436628218139958335e-01 3.42152968218706818315233419980359e-02 3.64002553476138815402585180436290e-02 3.98777051504644892282458457088978e-03
            3.75457566621084440350308852885064e-02 9.25786888466940460773457743925974e-01 3.66673548709510743748296590638347e-02 4.17752614547965955144048777469834e-03
            8.68957390638383855852033832434245e-03 1.58497125150980022745272890460910e-01 8.32813300942636169921229338797275e-01 4.80833304318707948493694104286078e-03
            1.54759705396456059833809604242560e-01 8.36360665768839717237881359324092e-01 8.87962883470427843946026769117452e-03 4.81591289252254316916035392637241e-03
            8.33102529418505488578716722258832e-01 8.92572448244858054899886212751881e-03 1.57971746099045934341731367567263e-01 4.92887303791674120934107605762620e-03
            8.37423107352594398022915811452549e-01 1.52916730407855333551125909252733e-01 9.66016223955026842595827929471852e-03 5.13289401506833355831593834750493e-03
            1.55936250523392300193492587823130e-01 9.49662400580272500483225428524747e-03 8.34567125470804915821076974680182e-01 5.15940515555491762550444079238332e-03
            9.85996420952242602631088175257901e-03 8.34221149359543612256118194636656e-01 1.55918886430933989473146539239679e-01 5.31455008152322341818951656478021e-03
            4.05587373328940847816426185090677e-01 7.43893020079156377932783072992606e-03 5.86973696470267647384844167390838e-01 5.34406534476826889507039552995593e-03
            5.96472789861799967958688739599893e-01 3.95633080931094838206263375468552e-01 7.89412920710519383504788493155502e-03 5.34845105051586985900335946553241e-03
            8.07478004157673352747526251960153e-03 4.03131942590241598711742199157015e-01 5.88793277368181655617718206485733e-01 5.45132308568089651162891229319030e-03
            7.50739777207006658832355583399476e-03 5.85160959468056462995377842162270e-01 4.07331642759873435721829082467593e-01 5.49498917872476289048222497513052e-03
            3.93676451923723913495223314384930e-01 5.97489659289885399751085515163140e-01 8.83388878639068675369117045192979e-03 5.67115276145129746515616275814864e-03
            5.84653072621227898331142114329850e-01 8.72504649681958392715142025508612e-03 4.06621880881952502129195181623800e-01 6.02678214649227626809047109190942e-03
            4.87080411211974872998098362586461e-01 2.02129229911995021706783148829345e-02 4.92706665796825649117351986205904e-01 6.98095969102918278287317477293072e-03
            2.68351281178442480079837650919217e-01 7.20234008866831532102992241561878e-01 1.14147099547259323060188762610778e-02 7.05739957682016780943223821509491e-03
            7.22395628874778816808088777179364e-01 2.66239936645638886059117567128851e-01 1.13644344795822971327936556917848e-02 7.09651735229200816834627119078505e-03
            2.71682674235740162504981753954780e-01 1.12882698808235495957497107610834e-02 7.17029055883436350349313670449192e-01 7.21063381341644025340764656561987e-03
            1.12580842045890902602955918609950e-02 7.16969596332504788271933193755103e-01 2.71772319462906142284452926105587e-01 7.23521734274508269979531149829199e-03
            1.15034734369742069892694757982099e-02 2.74006711016551163240251298702788e-01 7.14489815546474615892691417684546e-01 7.24748849360278089154485670064787e-03
            7.14052590056392100237303566245828e-01 1.13511560497063096475400811868894e-02 2.74596253893901576237368544752826e-01 7.26933878470379171710114007964876e-03
            4.90287105311137894059214659137069e-01 4.93649184146848718501843222838943e-01 1.60637105420133319277908867661608e-02 7.29820954627874874526272108710145e-03
            2.01423425209245401945601372517558e-02 4.83257345960130813011801365064457e-01 4.96600311518944681488108017219929e-01 7.36572892331421608808073031582353e-03
            3.61107464858593657508478713680233e-02 9.35679501581967443257781269494444e-02 8.70321303355943931556737425125903e-01 8.37319816520742075949268468093578e-03
            8.60799881985092385505708989512641e-01 3.97379067075453590307532181213901e-02 9.94622113073622554635377923659689e-02 8.44777502287683522419481363385785e-03
            1.00589152600109102420056217397359e-01 8.58634341935195832817839800554793e-01 4.07765054646950231287405586044770e-02 8.47113314420841305441012281107760e-03
            9.18740717058512956771920698884060e-02 3.95513001973351110973631250544713e-02 8.68574628096813627919914324593265e-01 8.65350860477441877205428966135514e-03
            8.60488829619093564815557328984141e-01 9.66224057079417136817411915217235e-02 4.28887646729647215027014794941351e-02 8.72622732463379686962401393657274e-03
            4.39842178673168215929933921870543e-02 8.56188634910677426326230943232076e-01 9.98271472220056965696244333230425e-02 8.86086110796604174699719891350469e-03
            2.01101760673522950151337340685131e-01 7.44911583562643420108884129149374e-01 5.39866557638336574953541457944084e-02 1.41412012011636611941156616012449e-02
            7.44999372626340194969429830962326e-01 5.36865638165825095895478114016441e-02 2.01314063557077288502128453728801e-01 1.42498356244064438430596553075702e-02
            5.32186641309854921444433273336472e-02 1.96375427593497520195597871861537e-01 7.50405908275517008476640512526501e-01 1.42502823269358876168144689700057e-02
            7.45398464740035127107375956256874e-01 1.98206580555018274658607424498769e-01 5.63949547049465982340166192443576e-02 1.50323611738850931168753888300671e-02
            1.95728993287614227991610960089019e-01 5.55713833156082359643335166765610e-02 7.48699623396777536044055523234420e-01 1.51015638541140244377469770142852e-02
            1.09253205798743882604107113820646e-01 6.10003618241300005209382106841076e-01 2.80743175959956126064298587152734e-01 1.51993568038457978891520028241757e-02
            5.67625702000490311993630143660994e-02 7.40912189495913087178280420630472e-01 2.02325240304037867744568757188972e-01 1.52834398036998143910469849515721e-02
            4.83837933474769513186863889586675e-02 6.07513566097787727215973063721322e-01 3.44102640554735272893083219969412e-01 1.53033706501137598682271701022728e-02
            1.08061280976018553312556491619034e-01 1.12208151043696344184574797964160e-01 7.79730567980285171891807749489089e-01 1.54665034100298359742708953490364e-02
            6.18560590099037743705423508799868e-01 2.69875370303500006219366014192929e-01 1.11564039597462250075210477007204e-01 1.54886910417429742720107910258776e-02
            7.72129601349661354170450522360625e-01 1.11411739533305520488681850110879e-01 1.16458659117033125340867627528496e-01 1.56573125272326148238466458906260e-02
            6.11573480113249079437309774220921e-01 3.38936767793083126854014608397847e-01 4.94897520936677937086756173812319e-02 1.56786746695895332082137940687971e-02
            3.38132610337598116245771961985156e-01 4.94693938787471934603878764846741e-02 6.12397995783654724988309681066312e-01 1.57160234643337129445406219474535e-02
            1.17308412825418789671161334808858e-01 7.69645130979520919467518069723155e-01 1.13046456195060263105744979839073e-01 1.57591071946765054478944989568845e-02
            2.67455126059624903600564493899583e-01 1.11571880815410764964390466502664e-01 6.20972993124964234290530384896556e-01 1.62124068992757254525916010834408e-02
            6.54210016002574556992499310581479e-01 1.90654831469986768555102685240854e-01 1.55135152527438674452398004177667e-01 1.73756076192967817695134158384462e-02
            5.38297481157757953806886064285209e-02 3.35861682684902573825524996209424e-01 6.10308569199321637732680301269284e-01 1.75196727463413602643438338191118e-02
            1.84884032411679638530088709558186e-01 1.55183152385134193185223239197512e-01 6.59932815203186251551414898131043e-01 1.75358710155160517352168625393460e-02
            3.37626710474410451912063990675961e-01 6.08140259629467938573554874892579e-01 5.42330298961216650255323656892870e-02 1.76064607667074657371131252148189e-02
            6.06710203449961738009221789980074e-01 5.42632795598224151878419263539399e-02 3.39026516990215853741830187573214e-01 1.76307752490747621232358710585686e-02
            4.61261408549577756055271038349019e-01 6.88176670721689642817864296375774e-02 4.69920924378253279662942532013403e-01 1.83201610171350760813968605589253e-02
            1.52546536567121610028507916467788e-01 6.51024084574876482633953855838627e-01 1.96429378858001935093113843322499e-01 1.83866553835162795105251376526212e-02
            7.00582543543041696976203525082383e-02 4.66190439274151768866971679017297e-01 4.63751306371544103068771391917835e-01 1.85837831468433009596274985142372e-02
            4.70420137903155344272221327628358e-01 4.63482645535340020970238583686296e-01 6.60972165615045792463888574275188e-02 1.86685785803141907068791738311120e-02
            1.21646169374595991552467921792413e-01 2.38149487551563648901620240394550e-01 6.40204343073840331790336222184123e-01 2.01986673293932765538283291562038e-02
            6.37140405270220422373483870615019e-01 1.23839938451338629765707821661636e-01 2.39019656278440961738596115537803e-01 2.06790020318823875022928859834792e-02
            2.37990451511855216626400988388923e-01 6.37021645232653477286532961443299e-01 1.24987903255491361598217281425605e-01 2.10978895935022142671400047220231e-02
            1.48392985717712283522118354994745e-01 4.89418857778011029413534060950042e-01 3.62188156504276659308771968426299e-01 2.47725502018576737572885804183898e-02
            3.59806957154975348700531867507380e-01 1.45288086625325107981154815206537e-01 4.94904956219699543318313317286083e-01 2.50209630570553745454454031005298e-02
            4.94144105509497388872830470063491e-01 3.61021638381830600739164083279320e-01 1.44834256108672010388005446657189e-01 2.52897293557296266863776423861054e-02
            1.44063068798082477561806058474758e-01 3.51350834188701499627427438099403e-01 5.04586097013215995055190887796925e-01 2.60018605093971372221872684349364e-02
            5.01976444000366672781865418073721e-01 1.43549166329308974798806275430252e-01 3.54474389670324352419328306496027e-01 2.60766783942758678671580696573074e-02
            3.55542383429807595796035002422286e-01 5.01649159950201317670348544197623e-01 1.42808456619991086533616453380091e-01 2.62449576178954320504477237818719e-02
            2.44343954077136160663030750583857e-01 2.40605212910407650905852960931952e-01 5.15050833012456132919965057226364e-01 2.99579881257902615188104533672231e-02
            2.43706498934185139537689224198402e-01 5.10901727705534658419139759644167e-01 2.45391773360280174287595400528517e-01 2.99804998713138898258634412741230e-02
            5.12220080732083804520016201422550e-01 2.45273797354294331718449484469602e-01 2.42506121913621863761534314107848e-01 2.99957636064387064833702822852501e-02
            2.52603831517775601067654633880011e-01 3.70031955509367738166304206970381e-01 3.77364212972856605254889927891782e-01 3.17066591724676841157481987920619e-02
            3.75989565285068993993888852855889e-01 2.50540661163056177507257871184265e-01 3.73469773551874828498853275959846e-01 3.17655930554187962422396651618328e-02
            3.72907798714407878737375767741469e-01 3.75375027754923540346254640098778e-01 2.51717173530668636427520823417581e-01 3.18603302836001217590577994087653e-02
            ];
        
        
    otherwise
        xyw=[];
end




pointset_stats=[
    13   36 5.69e-16    0    0
    16   55 7.82e-13    0    0
    18   66 6.28e-16    0    0
    20   78 6.59e-16    0    0
    ];

