
function [xyw_bar,pointset_stats]=set_taylor_wingate_bos_siam_barycentric(deg)

%--------------------------------------------------------------------
% CUBATURE RULE ON THE SIMPLEX [0 0; 0 1; 1 0] WITH ADE EQUAL TO deg.
% THE CUBATURE RULE IS DESCRIBED IN BARYCENTRICAL FORM.
%--------------------------------------------------------------------
% INPUT:
% deg: algebraic degree of precision (ADE) of the rule.
%--------------------------------------------------------------------
% OUTPUT:
% xyw_bar: MATRIX OF NODES xyw(:,1:3) IN BARYCENTRICAL COOR. AND WEIGHTS xyw(:,4).
%      REMARK: The sum of the weights makes 1.
%
% pointset_stats:
%    statistics of the pointset stored in a matrix. Each rows contains
%    degree, cardinality of the full pointset, moment error w.r.t.
%    orthn. Dubiner basis, number of points with negative weights,
%    number of points outside the domain.
%--------------------------------------------------------------------

% SOURCE:
% M.A. Taylor, B.A. Wingate, L.P. Bos,
% "A cardinal algorithm for computing multivariate quadrature points",
% SIAM J. Numer. Anal.,
% Vol. 45, no.1, (2007),  pp.193-205.
%--------------------------------------------------------------------
% | DEG | CARD | W- | PT.O|   M.E.   |Q.|
%--------------------------------------------------------------------
% |  13 |  36 |   0 |   0 | 3.94e-13 |PI|
% |  16 |  55 |   0 |   0 | 7.82e-13 |PI|
% |  18 |  66 |   0 |   0 | 5.01e-13 |PI|
% |  20 |  78 |   0 |   0 | 7.68e-13 |PI|
%--------------------------------------------------------------------
% LEGEND:
%--------------------------------------------------------------------
% DEG  : algebraic degree of precision
% CARD : cardinality of the pointset
% W-   : negative weights
% PT.O : points outside the simplex [0 0; 1 0; 0 1].
% M.E. : maximum moment error in inf norm.
% Q.: quality of the rule.
%        PI: pos. weights, points in simplex
%        NI: some neg. weights, points in simplex
%        P0: pos. weights, some points out of the simplex
%        N0: some neg. weights, some points out of the simplex
%--------------------------------------------------------------------




switch deg
    
    case  13
        
        % ALG. DEG.:   13
        % PTS CARD.:   36
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.9e-13
        
        xyw_bar=[
            2.42935351589999987131918146587850e-02 9.49305929384600011289307985862251e-01 2.64005354563999761197123916645069e-02 8.31204993784999922323830645609632e-03
            2.65193427721999996915513975181966e-02 2.42695130639999992472333190107747e-02 9.49211144163799924733382340491517e-01 8.34058498889999992698118802536555e-03
            9.49212602355099965834028807876166e-01 2.65067966436999986890032943165352e-02 2.42806010012000354769678978072989e-02 8.34152845335000073245801388566179e-03
            3.37757637490000012236990301062178e-03 4.76731641236300018871219208449475e-01 5.19890782388799932434153561189305e-01 8.78404350414999958440720462249374e-03
            4.75767229810100000797490338300122e-01 5.19892182910200006951129125809530e-01 4.34058727969999225138053589034826e-03 9.22373309225000062583355031620158e-03
            5.19078319347100025638042097853031e-01 5.59127062019999961084426587376583e-03 4.75330410032699990363624920064467e-01 9.89712050940000018228737133085815e-03
            8.61683974532099994370071271987399e-01 1.33996048618000001412253752164361e-02 1.24916420606100003753979876819358e-01 1.01770197927500000156797455019841e-02
            1.24920975992600005732846568662353e-01 8.61305432133400050709326478681760e-01 1.37735918739999574356147604703438e-02 1.03426431969999999688214487036930e-02
            1.38565453861000005553183456186161e-02 1.24773371735799998294424995037843e-01 8.61370082878099907475188956595957e-01 1.04135683042999999114996967364277e-02
            2.11887064221999987800870712817414e-02 8.43843835122299990025851457176032e-01 1.34967458455499955682910240284400e-01 1.58909889139500007382288515600521e-02
            8.43229678721900022253521456150338e-01 1.35456364583000005152868538971234e-01 2.13139566950999725936100048784283e-02 1.60236017620500013902695712886270e-02
            1.35423179786500008647820436635811e-01 2.13482820656000008008312818219565e-02 8.43228538147899997490242185449461e-01 1.60303840572999997671210792304919e-02
            3.08885351067899993626753030184773e-01 2.21919663013999991785230037066867e-02 6.68922682630700093930897764948895e-01 2.15382979591500003935244933472859e-02
            6.68505759516900055494659227406373e-01 3.08901287938899982155049883658648e-01 2.25929525441999623502908889349783e-02 2.19236707669499993056394515633656e-02
            2.26545012556999998676321439461390e-02 6.69170994332100033119559157057665e-01 3.08174504412199956604467843135353e-01 2.19604836366499994726453337534622e-02
            2.80851540877199989942880620219512e-01 6.92471815510599997267604521766771e-01 2.66766436122000127895148580137175e-02 2.39975961845499997926456359209624e-02
            6.92244674905099999229207696771482e-01 2.68723345026000010049571642412047e-02 2.80882990592299985888047331172857e-01 2.41903130366500004888585806384071e-02
            2.68617447119000006539035751984557e-02 2.81009397322199983015877933212323e-01 6.92128857965900068371922770893434e-01 2.42433711687499997555761410694686e-02
            1.14177848546999999301831962839060e-01 7.97358141358599947778884597937576e-01 8.84640100944000806748590548522770e-02 2.78482244012000015875862857228640e-02
            7.97480792206100019825498748105019e-01 8.79806508790999947100530675925256e-02 1.14538556914799985464448184302455e-01 2.80513182177999989552752424515347e-02
            8.92807293894000014722323044225050e-02 1.14502056112799996401285795855074e-01 7.96217214497800029882057515351335e-01 2.82595061846500003599480521643272e-02
            1.05248789245500004585487374697550e-01 6.68690411992200028912236575706629e-01 2.26060798762300008135639473039191e-01 3.44644945334999994179803195493150e-02
            6.66302228073999969204521676147124e-01 2.27505163183199993204297584270535e-01 1.06192608742800037591180739582342e-01 3.58606668044499990455342697259766e-02
            2.30780373754700007804530059729586e-01 1.05457256122099995998553367826389e-01 6.63762370123199940685765341186197e-01 3.63726960488000017335252778138965e-02
            1.70505915754000009210145094584732e-01 5.17406439865800038546694850083441e-01 3.12087644380199979998735670960741e-01 3.94403668368499965568574339158658e-02
            5.08659397304299987219167178409407e-01 3.17052385520899981941056466894224e-01 1.74288217174800030839776354696369e-01 4.05057172755999977953322854773432e-02
            3.14182386228100019032183354283916e-01 1.81070636165899995217998252883262e-01 5.04746977605999957994242777203908e-01 4.12862649527500000234958577038924e-02
            4.61746081786399975577239729318535e-01 4.67859453980400019634799946288695e-01 7.03944642332000047879603243927704e-02 4.21022283665000024432956138298323e-02
            6.93087496081000037362329635470815e-02 4.62285604208500011669258356050705e-01 4.68405646183400026227872103845584e-01 4.21792766652500025759131574432104e-02
            4.65195525926800024230090002674842e-01 7.24357805669000043291561041769455e-02 4.62368693506299943685178277519299e-01 4.25984934244000024161636019925936e-02
            2.57862585789299991745338047621772e-01 6.13139503917700001700552547845291e-01 1.28997910293000006554109404532937e-01 4.51422664026000022841067504941748e-02
            6.11262776677899966060181213833857e-01 1.30036083460900009356464579468593e-01 2.58701139861200024583354206697550e-01 4.57141571742500016362065196062758e-02
            1.30518213593399989846943753946107e-01 2.58171382888400025024111528182402e-01 6.11310403518199985128944717871491e-01 4.58139532704500002102676603499276e-02
            4.28143799182799988933822987746680e-01 2.36200596981700011722793419721711e-01 3.35655603835499971587807976902695e-01 5.12786687447999989242219953666790e-02
            3.35699578373000018238769825984491e-01 4.31102630858800017765730672181235e-01 2.33197790768199963995499501834274e-01 5.16579830706499973369538736278628e-02
            2.30542429883599991891074409977591e-01 3.45601394937600003842703699774574e-01 4.23856175178800032021797505876748e-01 5.17927183596499987605632497889019e-02
            ];
        
    case  16
        
        % ALG. DEG.:   16
        % PTS CARD.:   55
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 7.8e-13
        
        xyw_bar=[
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 3.10129992550000005771537736620758e-04
            0.00000000000000000000000000000000e+00 1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 3.15758735599999988078595558604889e-04
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00 1.00000000000000000000000000000000e+00 3.54330077950000001639485613935676e-04
            9.39886358357699958077091650920920e-01 4.98487446340000004557646562375339e-03 5.51287671789000427446936214437301e-02 2.75818580839999995604761018341833e-03
            5.43806683058000034369783293186629e-02 9.38640561861699973533745833265129e-01 6.97876983250000915148802960175090e-03 3.13462038279999988601987048753017e-03
            9.39400491640000022408596436207517e-03 5.26424462697000009048231561337161e-02 9.37963548813899983258579595712945e-01 3.92657044129999990150770017294235e-03
            1.64345086361999993818105991749690e-02 9.46903551735099990871447062090738e-01 3.66619396287000132161892906879075e-02 4.72757419320000028123418545078493e-03
            9.46948726986200028932216810062528e-01 3.63373677167000003196406510141969e-02 1.67139052970999707481425389232754e-02 4.89122556354999998146082518246658e-03
            4.26604005767999977516424792156613e-02 1.51224541798999998626351626285214e-02 9.42217145243300024937127545854310e-01 4.99308217444999983408715493737873e-03
            1.22269495439000005360119516240047e-02 8.69377351066400017032265168381855e-01 1.18395699389699959880317692295648e-01 6.87769094080000007590536270640769e-03
            8.67369652104699961903122584772063e-01 1.20491728577399995203656146713911e-01 1.21386193179000428932212685140257e-02 7.04895890199999999858881949421630e-03
            8.45674402138899994341159072064329e-01 1.57763967870000014481757233397730e-02 1.38549201074100014619006060456741e-01 7.48234321684999956897366146790773e-03
            1.39575963210299996974939062965859e-01 8.44812087037499992447919794358313e-01 1.56119497522000383327167583047412e-02 7.80487518060000041303370110767901e-03
            1.31782174323099993129559948101814e-01 1.35009605583999992006738466443494e-02 8.54716865118499979914190589624923e-01 7.88418466740000052628278837119069e-03
            1.57955126300000012196100840355939e-02 1.45527493853600004003467915936199e-01 8.38676993516400015593603711749893e-01 8.78972731915000048130703902415917e-03
            7.36546288443600039030911830195691e-01 1.55697540907999993492483170598462e-02 2.47883957465599963354563328721269e-01 1.02056920135000003119207789836764e-02
            1.39688430330000001866608272393933e-02 7.37983689445000035078692235401832e-01 2.48047467521999909223495706100948e-01 1.04781439308000001026099923251422e-02
            2.54789518603899978188565000891685e-01 7.29761568977099983612788491882384e-01 1.54489124190000381986465072259307e-02 1.05356706499000005866628626449710e-02
            7.31638652255499954968342990468955e-01 2.54307668331499980851617692678701e-01 1.40536794130000641800393168523442e-02 1.08823380101000007774691979989257e-02
            1.57253728950999989433245929149052e-02 2.69623979579100003789449147006962e-01 7.14650647525799986858885404217290e-01 1.11144204349499994832761728957848e-02
            2.66230284364699976151769078569487e-01 1.44783956308000005819280886498746e-02 7.19291320004500023266302832780639e-01 1.12093346841000002234167709502799e-02
            8.67350406521400052461956420302158e-01 5.91679410399999980429974755224976e-02 7.34816524385999425561522002681158e-02 1.15061308496499994924455734235380e-02
            7.41493666957000058248539176020131e-02 8.63478257506099966533952283498365e-01 6.23723757981999860078303754562512e-02 1.18406951249999999736806088890262e-02
            1.59285948359999991741364766539846e-02 4.19123895523800027440586291049840e-01 5.64947509640200018488087607693160e-01 1.28732321684000006190018439156120e-02
            1.56061028068000002405613102496318e-02 5.80922292114600002932434108515736e-01 4.03471605078600026317303672840353e-01 1.28978400803999998514015601358551e-02
            5.91009481748399956302364444127306e-01 1.59251452650999991700331293031923e-02 3.93065372986500016772026810940588e-01 1.29036163804999992610333947595791e-02
            4.03477149688900027513227541930974e-01 5.80670036810399969873230929806596e-01 1.58528135007000026135415282624308e-02 1.30171616029500000072394172434542e-02
            5.69474562852600008078240989561891e-01 4.14949514630200022580197583010886e-01 1.55759225171999693415614274272230e-02 1.32884070804499994367109394488580e-02
            6.78493700649999975649251382492366e-02 7.61218678590999953126328136931988e-02 8.56028762075899951611290816799738e-01 1.32892380915500003119023020303757e-02
            4.26596859027200014935488070477732e-01 1.57509692312000001979122032480518e-02 5.57652171741599977927705822366988e-01 1.33766164619000000463033828168591e-02
            6.70982507890000023920151761558373e-02 7.74189831242099946173595981235849e-01 1.58711917968900051434388842608314e-01 1.87893903320500016418392874584242e-02
            7.52831023147999989220124916755594e-01 8.19119495638999939313151799069601e-02 1.65257027288100016848559903337446e-01 1.91532947097499996191860560657005e-02
            7.75372778355699954211388558178442e-01 1.57712845729199996336333811086661e-01 6.69143759151000494522776307348977e-02 1.92424847512500003798852077352421e-02
            1.68907315778699995689748902805150e-01 7.50394309974199980040054924756987e-01 8.06983742471000242701961724378634e-02 1.94809912925999993704717638820512e-02
            1.68733583291899996714491294369509e-01 7.08311507268000045689504418078286e-02 7.60435265981299957083194840379292e-01 1.97302055773499983304386518057072e-02
            8.21244708436000003803911795330350e-02 1.76299662677100010821362729984685e-01 7.41575866479299961042670474853367e-01 2.06182389048999992198218933481257e-02
            6.28870536334499963260213917237706e-01 8.07744953317000025094785087276250e-02 2.90354968333800034230307574034669e-01 2.56436219241500014398571494211865e-02
            8.11413015265999987279954552832351e-02 3.05437358977599993092866270671948e-01 6.13421339495799911034623619343620e-01 2.58202820967500003279671005884666e-02
            2.96911206508000014014925227456843e-01 6.22748598887099991117111130733974e-01 8.03401946049000503791148730670102e-02 2.59115021134499988464394704124061e-02
            7.67542314170999995237565372008248e-02 6.24724714954600024441333516733721e-01 2.98521053628299948279334330436541e-01 2.64263994090499984568332081380504e-02
            6.22302233384499969837122534954688e-01 3.01148582116600005420536945166532e-01 7.65491844989000247423405198787805e-02 2.69252786513499986487119031153270e-02
            3.10378628805099976428749641854665e-01 7.79098365078999960386951784130360e-02 6.11711534687000013654767371917842e-01 2.70947664659500001005376645935030e-02
            8.19218215187000053933985554976971e-02 4.60363303835099990646284595641191e-01 4.57714874646200031715892464490025e-01 2.92368573222000008304810592107970e-02
            4.71702266501299982692785306426231e-01 8.21554006796999941597547945093538e-02 4.46142332819000009269672091249959e-01 2.96431584181500000496622959644810e-02
            4.54660341525000011309032288409071e-01 4.63756503388999985482143983972492e-01 8.15831550860000032088237276184373e-02 2.97179138374500002861555714162023e-02
            1.70109133923700001078671562027012e-01 6.42227780818800053630468482879223e-01 1.87663085257499973046435570722679e-01 3.15900127931499977163021242176910e-02
            6.40600432948699949342596937640337e-01 1.89829353725600002977813574034371e-01 1.69570213325700047679589488325291e-01 3.16463422576499994742782462253672e-02
            1.91226758371700000793680374044925e-01 1.73995568534300010243853762403887e-01 6.34777673094000016718041479180101e-01 3.20353680885999969607524917591945e-02
            1.88531576706999992598667859056150e-01 4.79891407040600015410092282763799e-01 3.31577016252400047502391089437879e-01 4.06020297959000034393461930903868e-02
            4.77292995769100003577989355108002e-01 3.34835659811899999382944770331960e-01 1.87871344418999997039065874560038e-01 4.07218756764999995079179484491760e-02
            3.12697462175999973954532151765306e-01 4.95797219725900017994746349359048e-01 1.91505318098100063561872730133473e-01 4.07339600620499966976417738351302e-02
            4.96122594594600008210960595533834e-01 1.92755366890399987278570392845722e-01 3.11122038515000087777195858507184e-01 4.07525274041999965857208110264764e-02
            1.92880531286699991033728451839124e-01 3.16101580726100006568657363459351e-01 4.91017887987199974642038569072611e-01 4.07582332469499999949569257751136e-02
            3.36004145381600005659805674440577e-01 1.89489280128999987296012363913178e-01 4.74506574489399923777455114759505e-01 4.08465529811500027701498538590386e-02
            3.33728055084800023788460521245725e-01 3.34357102181099985482859437979641e-01 3.31914842734099935217528809516807e-01 4.61609167265499970023157061405072e-02
            ];
        
    case  18
        
        % ALG. DEG.:   18
        % PTS CARD.:   66
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.0e-13
        
        xyw_bar=[
            1.16731059668000007600863199286323e-02 9.81256595128899955504664376348956e-01 7.07029890429999863243892832542770e-03 1.25828784929999997962979385590643e-03
            9.81003085838800026330375203542644e-01 7.14625048630000009319829246123845e-03 1.18506636748999735764265039961174e-02 1.26367260034999993704996068544233e-03
            1.06966317092000006566676617580924e-02 1.15153933376000005422978844649151e-02 9.77787974953200023087163117452292e-01 1.66346476664999990899285187140322e-03
            9.38247698355100023803743169992231e-01 4.95570591341000024621088471121766e-02 1.21952425107999737341479828955926e-02 4.07517460624999983825889060540248e-03
            1.26627518416999994971261500609216e-02 9.37012362061499959331456466316013e-01 5.03248860968000810700573310896289e-02 4.30677628710000003076086017017587e-03
            5.98109409984000015003680061909108e-02 1.21364578921999994448599480278972e-02 9.28052601109400021606177233479684e-01 4.38933730895000023125573918036935e-03
            1.37363297926999996384500590806965e-02 6.12783625596999975693179862901161e-02 9.24985307647600007996402382559609e-01 4.85497927810000021175973472509213e-03
            9.22952795940499948201818369852845e-01 1.41128270601999991629149633354245e-02 6.29343769993000456963727629045025e-02 5.12331059575000039019165853915183e-03
            6.33107354992999971088352140213829e-02 9.22019729172700031050169400259620e-01 1.46695353279999718409953857189976e-02 5.41988441705000024983807094258736e-03
            1.17265100335000004017071972839403e-02 1.50052047522899995035672304766194e-01 8.38221442443599973337597930367338e-01 6.46926950880000010968284840373599e-03
            1.55472058732299994732173331613012e-01 8.32514712158900049843168744700961e-01 1.20132291087999831802335393149406e-02 6.81699117915000006584502756368238e-03
            8.34329388898199963975343962374609e-01 1.25228158758999997407501680868336e-02 1.53147795225900024140841537700908e-01 6.92386640734999996016352241667846e-03
            8.50163803195699951942287952988409e-01 1.37199750873600007095021169334359e-01 1.26364459307000409626908776772325e-02 6.97107700525000009011966639604907e-03
            1.28816350522000001854738826523317e-02 8.47762706347899963610359463928035e-01 1.39355658599900067429189221002162e-01 7.20606999839999966955828014647523e-03
            1.51080160895900011963632891820453e-01 1.36526924038999995097976025704156e-02 8.35267146700199969444611269864254e-01 7.68517277669999998579442035406828e-03
            1.01917879217000000718895691420585e-02 5.77043861834499960394850859302096e-01 4.12764350243800093309687326836865e-01 8.12449011265000027581795194464576e-03
            2.81337239930300020418485473783221e-01 7.06685375962299966445812060555909e-01 1.19773841074000131357024656608701e-02 8.48591521400000013686248934163814e-03
            7.12437462850100011735321459127590e-01 1.24569780989999996656525382832115e-02 2.75105559050900005946260762357269e-01 8.50442662105000031502743951250523e-03
            2.76302525086299988199556310064509e-01 1.21741311386000007621621676889845e-02 7.11523343775099958996577242942294e-01 8.54767603374999990206717370710976e-03
            1.09658368560999995577898147303131e-02 4.19430671246599995338755206830683e-01 5.69603491897300084900734873372130e-01 8.69444272795000076825555623827313e-03
            4.28911051788399999562528819296858e-01 5.59961606746899986930543491325807e-01 1.11273414646999579957764581195079e-02 8.72719812195000069210415460929653e-03
            4.21542055511499980102740892107249e-01 1.16475994784999998393759668147140e-02 5.66810345010000049548182232683757e-01 8.92033786434999996506345354418954e-03
            5.71125859044400052688672531076008e-01 1.18218313988999994962325601477460e-02 4.17052309556699973835947048428352e-01 8.92234319394999940422419371088836e-03
            5.82686827051099975882664239179576e-01 4.05788958117699993799476487765787e-01 1.15242148312000303178592730546370e-02 8.95231687759999970877000663449508e-03
            1.30567806713000004320557323467256e-02 2.72502375086799975001383700146107e-01 7.14440844241900019362390139576746e-01 9.06298781005000081423261804047797e-03
            1.30760400963999996737374686972544e-02 7.22471252323300006104034309828421e-01 2.64452707580300039325038596871309e-01 9.23924194410000003407912316788497e-03
            7.26343706240699993337273099314189e-01 2.60298401925100009179914195556194e-01 1.33578918341999974828127051296178e-02 9.28967821855000014386050111170334e-03
            6.87230068637000035547401921576238e-02 6.31417277210000044140869590592047e-02 8.68135265415299950397809425339801e-01 1.01608575888500005796055702944614e-02
            8.65230210152900025555311458447250e-01 7.20611837337999938668531285657082e-02 6.27086061132999805778354129870422e-02 1.06885830904499996851786391971473e-02
            6.48599071037000013406625953393814e-02 8.59043354390999991387900536210509e-01 7.60967385052999656380734450067393e-02 1.15958427048999997838185649357001e-02
            1.48349494336200010691229067560926e-01 7.88878835224000041215219880541554e-01 6.27716704397999203379754362686072e-02 1.37213355429500003662690588157602e-02
            6.24359898395999979303638838246115e-02 1.49393549935399994055273964477237e-01 7.88170460225000035769937767327065e-01 1.45150961169999998673363350576437e-02
            7.87136901173499992090398791333428e-01 6.56382042756999961596875436953269e-02 1.47224894550800011749913664971245e-01 1.47261369252500006615802519149838e-02
            5.19104921610000005771645703589456e-02 5.25563569560499987964874435419915e-01 4.22525938278500046152430513757281e-01 1.49718125814500002679352519407985e-02
            1.54312992744399996025705945612572e-01 7.16383926916999980161193661842844e-02 7.74048614563899950447023456945317e-01 1.53513474059500006074774347553102e-02
            2.61784274560300012080205078746076e-01 6.21479485287999983134099579729082e-02 6.76067776910899920217445924208732e-01 1.62631682931499987887491442961618e-02
            7.66725787281300030784336740907747e-01 1.65821155483100002303586961716064e-01 6.74530572355999669120762973761884e-02 1.63942104252999994173922715390290e-02
            2.58210367662700002178155500587309e-01 6.80011976613900004373647334432462e-01 6.17776557233999934481971649802290e-02 1.65617337595999995769613377660789e-02
            6.79065925147000065686953007570992e-02 7.57151543778200042389414647914236e-01 1.74941863707099964919677859143121e-01 1.73083763437500010029523878074542e-02
            5.29357827480399945052624843810918e-01 4.12150384110699985473758033549530e-01 5.84917884089000694736171226395527e-02 1.73540686988000007751065112415745e-02
            6.66036150483999972449211668390490e-02 2.61251308788599978516486999069457e-01 6.72145076163000121383106488792691e-01 1.73686024702000008090863047982566e-02
            5.85675461898999966114942594685999e-02 3.90223611453499985568527108625858e-01 5.51208842356599948431039592833258e-01 1.74264381226999984231262885714386e-02
            6.44535360411000041169771179738746e-02 6.37362655976099978794024991657352e-01 2.98183807982799975455634466925403e-01 1.74300780593000005647130734587336e-02
            6.74813842915100048358567619288806e-01 6.37583342060999980205338033556473e-02 2.61427822878799953620898577355547e-01 1.77735784987500013143435495521771e-02
            3.91460231036899974910170385555830e-01 5.50323809056299961817160237842472e-01 5.82159599068000632726693766016979e-02 1.80091498191499999048659219624824e-02
            6.48770149230699977316305648855632e-01 2.83672836026300001499578229413601e-01 6.75570147430000211841161217307672e-02 1.81463142921499988313716045240653e-02
            3.94649822040800013489558750734432e-01 6.05175522554000019925801723275072e-02 5.44832625703799977578967173030833e-01 1.90948851041500000658146518617286e-02
            5.39013715193299991490505362889962e-01 6.11990176935999982688052512003196e-02 3.99787267113100031057371097631403e-01 1.96126400058999998676778631079287e-02
            1.62789508278500000271193925982516e-01 6.86132214103500004398483724799007e-01 1.51078277617999967574746733589564e-01 2.41355062943999984503840039451461e-02
            6.81243632264100007489560084650293e-01 1.56796834589900013767760356131475e-01 1.61959533145999978742679559218232e-01 2.44956060783000015290689077573916e-02
            1.54283287802000013133607581039541e-01 1.66751262402000000051316419558134e-01 6.78965449796000042326227230660152e-01 2.48610416936000007404050649029159e-02
            2.52272775044500019436810589468223e-01 2.50480393339500007066078524076147e-01 4.97246831616000029008262117713457e-01 2.53532868493000010379212483258016e-02
            2.54798153240699987165385209664237e-01 4.99409064904299981879631786796381e-01 2.45792781855000030954983003539382e-01 2.54885997021499985382142483558710e-02
            1.48558054919399995519313506520120e-01 5.75602309608699957976796213188209e-01 2.75839635471900046503890280291671e-01 2.60680031833500011617665848007164e-02
            2.93023960643599978848072851178586e-01 5.65689735416200045747814328933600e-01 1.41286303940199919892961588629987e-01 2.61730437462499992973263829298958e-02
            2.80899127231000023119378283809056e-01 1.43792157424800004505982542468701e-01 5.75308715344200027885790404980071e-01 2.62220341775999989109013910137946e-02
            4.82098959297100004839364828512771e-01 2.51855753586499975238410797828692e-01 2.66045287116400019922224373658537e-01 2.63729822411499989498473439653026e-02
            5.64187824544399973980546292295912e-01 1.46296674315299995994266168963804e-01 2.89515501140300002269611923111370e-01 2.64724531863999995584535440684704e-02
            1.30769964434400004504865933085966e-01 4.48957758611700008533063055438106e-01 4.20272276953899959206495395847014e-01 2.71197797250499993138017629235037e-02
            1.47969222194799993053138109644351e-01 3.00117438682899972945961053483188e-01 5.51913339122300006245325221243547e-01 2.71735101709499986266660442879584e-02
            5.63868422294599991317909370991401e-01 2.81377208929800004533205992629519e-01 1.54754368775600004148884636379080e-01 2.73550274319500005826277799769741e-02
            4.36115742879000012344192782620667e-01 4.25205344641999993804404311958933e-01 1.38678912478999993851402905420400e-01 2.78644172956499999727064675880683e-02
            3.60326393528499988594404612740618e-01 2.59919000488899987200142049914575e-01 3.79754605982600079716604568602634e-01 2.88867132116500009708648377682039e-02
            4.22418833467400001158154054792249e-01 1.45323844330299989380250735848676e-01 4.32257322202300064972746440616902e-01 2.92696890811500014251134160758738e-02
            3.71900183305200027206183222006075e-01 3.78012270356699986439963367956807e-01 2.50087546338099986353853410037118e-01 3.04519625339999995550233080621183e-02
            2.41364500692799993641912692510232e-01 3.84756328493999988360485531302402e-01 3.73879170813199990242026160558453e-01 3.18636982224499987204069384461036e-02
            ];
        
    case  20
        
        % ALG. DEG.:   20
        % PTS CARD.:   78
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 7.7e-13
        
        xyw_bar=[
            8.94113371120000034741881478339565e-03 8.69832937020000029892941029174835e-03 9.82360536918600035782844770437805e-01 1.08722726994999992558565971023654e-03
            9.79262262980699960124297831498552e-01 1.02644133744000005947771825276504e-02 1.04733236449000392809249859737974e-02 1.44935676325000004603660297419765e-03
            1.05475382112000005963015070165056e-02 9.78551420251499992630783708591480e-01 1.09010415372999824867861207167152e-02 1.54230146684999989276088072642779e-03
            2.37770619469999987607988778393064e-03 6.36551098604000026304206016902754e-02 9.33967183944899970171604763891082e-01 1.72008165519999998115119321795419e-03
            6.30425115794999990637137443627580e-02 4.15063475090000043310478616831460e-03 9.32806853669599922440625050512608e-01 2.09492360060000012630943011515683e-03
            9.30842249672999999532407855440397e-01 4.80534822629999965742930712053749e-03 6.43524021007000068816950033578905e-02 2.23690257489999981543871143685465e-03
            6.29076555490000050019361310660315e-02 9.31679006948100019158687246090267e-01 5.41333750289996196158881502924487e-03 2.35272104070000011993846200653024e-03
            9.31596224638099990045247977832332e-01 6.26264881800999984040601020751637e-02 5.77728718180001155069192009250401e-03 2.44339678750000012866228082941689e-03
            6.19516894150000038687942094384198e-03 9.29358705856400013622931055579102e-01 6.44461252021000241541059949668124e-02 2.59638216844999998589305079121914e-03
            2.87125819236999993422365662354423e-02 3.10202122996999993254085836724698e-02 9.40267205776600056843506081349915e-01 3.70365294905000003167216959809593e-03
            9.29384447830500004528175850282423e-01 3.42152968218999986582673500379315e-02 3.64002553475999968135567996796453e-02 3.98777051504999990177990909501204e-03
            3.75457566620999994011498301915708e-02 9.25786888466900048655361388227902e-01 3.66673548709999241879131659516133e-02 4.17752614550000005155805382628387e-03
            8.68957390639999924242253825923399e-03 1.58497125151000006759716143278638e-01 8.32813300942599976650626558694057e-01 4.80833304319999970538601630209996e-03
            1.54759705396499996910009144812648e-01 8.36360665768799971253599778719945e-01 8.87962883470005959196669209632091e-03 4.81591289249999957022829732977698e-03
            8.33102529418500048485896058991784e-01 8.92572448239999961805413164483980e-03 1.57971746099099946691879381432955e-01 4.92887303790000026043616188076157e-03
            8.37423107352599949138038937235251e-01 1.52916730407799988933348345199192e-01 9.66016223960006192861271756555652e-03 5.13289401505000013326007746172763e-03
            1.55936250523399988487938117032172e-01 9.49662400579999975425149472130215e-03 8.34567125470800030839768623991404e-01 5.15940515554999968445004654427066e-03
            9.85996420949999952121345359046245e-03 8.34221149359600011585769152588909e-01 1.55918886430900016648593009449542e-01 5.31455008149999980765487705980377e-03
            4.05587373328899991609119979329989e-01 7.43893020079999973959150594282619e-03 5.86973696470300065897163221961819e-01 5.34406534474999965478314578604113e-03
            5.96472789861799967958688739599893e-01 3.95633080931100000743327882446465e-01 7.89412920710003129798337795364205e-03 5.34845105049999974128338564582918e-03
            8.07478004159999963873506345635178e-03 4.03131942590299996442837482391042e-01 5.88793277368099943203105794964358e-01 5.45132308570000015390810688131751e-03
            7.50739777210000010926327362881239e-03 5.85160959468099983737943148298655e-01 4.07331642759800049979901359620271e-01 5.49498917874999964761073556474003e-03
            3.93676451923699988189042642261484e-01 5.97489659289900054695010567229474e-01 8.83388878640001262709802176686935e-03 5.67115276144999989199613210644202e-03
            5.84653072621200031733224022900686e-01 8.72504649680000063383111807979731e-03 4.06621880881999964163497907065903e-01 6.02678214650000012436725782549729e-03
            4.87080411212000019549606122382102e-01 2.02129229911999983015924442497635e-02 4.92706665796800058476634376347647e-01 6.98095969105000033194663444646721e-03
            2.68351281178499989632513234028011e-01 7.20234008866800001769092887116130e-01 1.14147099546999530872426475980319e-02 7.05739957679999991430053185581528e-03
            7.22395628874800022067859117669286e-01 2.66239936645599972742104455392109e-01 1.13644344796000051900364269386046e-02 7.09651735230000003940009634106900e-03
            2.71682674235699972431490323288017e-01 1.12882698808000007245633256047768e-02 7.17029055883500077150927154434612e-01 7.21063381340000027902581436478613e-03
            1.12580842045999999362360100008118e-02 7.16969596332500014312927305581979e-01 2.71772319462900036057817487744614e-01 7.23521734274999977348796775800110e-03
            1.15034734370000005926337749428967e-02 2.74006711016600013053334805590566e-01 7.14489815546400008905436607165029e-01 7.24748849360000012981286587887553e-03
            7.14052590056399982820778404857265e-01 1.13511560497000004582579535394871e-02 2.74596253893900021925134069533669e-01 7.26933878469999961158265477934037e-03
            4.90287105311200011037442436645506e-01 4.93649184146799979711062178466818e-01 1.60637105420000092514953848876758e-02 7.29820954629999997520517496241155e-03
            2.01423425209000007962689693385983e-02 4.83257345960100004322868016970460e-01 4.96600311519000026105885581273469e-01 7.36572892330000002919510038168482e-03
            3.61107464858999999135491520974028e-02 9.35679501582000056059129633467819e-02 8.70321303355900077747264731442556e-01 8.37319816519999961246245590018589e-03
            8.60799881985100046044578903092770e-01 3.97379067074999994813033765694854e-02 9.94622113073999614130116242449731e-02 8.44777502290000072149300791579662e-03
            1.00589152600099998591254291113728e-01 8.58634341935200051665333376149647e-01 4.07765054646999081100489092932548e-02 8.47113314419999964555163529666970e-03
            9.18740717058000033734543876562384e-02 3.95513001973000002942093544788804e-02 8.68574628096900003271230161772110e-01 8.65350860475000080440644012469420e-03
            8.60488829619100004109100154892076e-01 9.66224057078999970515909012647171e-02 4.28887646729999988393089438432071e-02 8.72622732464999918688963731483454e-03
            4.39842178673000017141703210654669e-02 8.56188634910699963853630833909847e-01 9.98271472220000344321988450246863e-02 8.86086110795000075901484137830266e-03
            2.01101760673499996290303215573658e-01 7.44911583562600010388621285528643e-01 5.39866557639000488322267301555257e-02 1.41412012011500002467423442453764e-02
            7.44999372626300004895938400295563e-01 5.36865638166000025410795615243842e-02 2.01314063557099992562982038180053e-01 1.42498356243999993453464014692145e-02
            5.32186641310000013715963973481848e-02 1.96375427593499990441827662834839e-01 7.50405908275500022064363747631432e-01 1.42502823269499995922915402957187e-02
            7.45398464740099964132014065398835e-01 1.98206580555000011489852340673679e-01 5.63949547049000243781335939274868e-02 1.50323611738999995957044575334294e-02
            1.95728993287599989381320142456389e-01 5.55713833155999994972695787964767e-02 7.48699623396800073571455413912190e-01 1.51015638540999991984437045289269e-02
            1.09253205798800004378001915483765e-01 6.10003618241300005209382106841076e-01 2.80743175959899948779252554231789e-01 1.51993568038499993894108186509584e-02
            5.67625702000999973750872129585332e-02 7.40912189495899986546589843783295e-01 2.02325240304000009139429039350944e-01 1.52834398037000000064589144699312e-02
            4.83837933475000023242351687713381e-02 6.07513566097800050691546402958920e-01 3.44102640554699967800900139991427e-01 1.53033706501000000416157220683999e-02
            1.08061280975999998710257443690352e-01 1.12208151043699994042768253166287e-01 7.79730567980300048880337726586731e-01 1.54665034100500003999556497547019e-02
            6.18560590099100027217104980081785e-01 2.69875370303500006219366014192929e-01 1.11564039597399966563529005725286e-01 1.54886910417499999020884970946099e-02
            7.72129601349699989931707477808231e-01 1.11411739533299997129134339957091e-01 1.16458659117000012939158182234678e-01 1.56573125272500002225228854513261e-02
            6.11573480113300038674140068906127e-01 3.38936767793100002243988910777261e-01 4.94897520935999590818710203166120e-02 1.56786746696000005296678381228048e-02
            3.38132610337600003624913824751275e-01 4.94693938787000020429474034244777e-02 6.12397995783700022087714387453161e-01 1.57160234643499985285330922124558e-02
            1.17308412825399999146469554034411e-01 7.69645130979500047274655116780195e-01 1.13046456195099898067724097927567e-01 1.57591071947000005426531288321712e-02
            2.67455126059599979093661659135250e-01 1.11571880815399995801051602484222e-01 6.20972993124999983471923314937158e-01 1.62124068992499995034428650342306e-02
            6.54210016002599981099763226666255e-01 1.90654831470000007964671340232599e-01 1.55135152527400010935565433101146e-01 1.73756076193000014162848287924135e-02
            5.38297481157999982426254348411021e-02 3.35861682684900020312568358349381e-01 6.10308569199299988383700110716745e-01 1.75196727463499991872541983184419e-02
            1.84884032411700011122590581180702e-01 1.55183152385099998316064784376067e-01 6.59932815203200018316920250072144e-01 1.75358710154999986041701731664944e-02
            3.37626710474400015815632514204481e-01 6.08140259629399992924447815312305e-01 5.42330298962000467710709017410409e-02 1.76064607666999994872725210370845e-02
            6.06710203449900009609052631276427e-01 5.42632795598000025605323060062801e-02 3.39026516990300008647096774438978e-01 1.76307752490500006803397781141030e-02
            4.61261408549600016026914772737655e-01 6.88176670721999950153247027628822e-02 4.69920924378199933446609293241636e-01 1.83201610171500016421841650071656e-02
            1.52546536567099988435103341544163e-01 6.51024084574900019362075909157284e-01 1.96429378858000047713971980556380e-01 1.83866553835000008654265712948472e-02
            7.00582543543000063612780081712117e-02 4.66190439274200008057391642068978e-01 4.63751306371499971703542541945353e-01 1.85837831468500004616917209432359e-02
            4.70420137903199975237811258921283e-01 4.63482645535299997430200846793014e-01 6.60972165614999718208366630278761e-02 1.86685785803000006688456835490797e-02
            1.21646169374600002233144380170415e-01 2.38149487551600008705676714271249e-01 6.40204343073799919672239866486052e-01 2.01986673294000003420212152605018e-02
            6.37140405270199994269830767734675e-01 1.23839938451299994004450866214029e-01 2.39019656278500025603506173865753e-01 2.06790020318999984150210025291017e-02
            2.37990451511899986369868997826416e-01 6.37021645232599964536746028898051e-01 1.24987903255500021337809357646620e-01 2.10978895935000007599846583161707e-02
            1.48392985717699987802120631386060e-01 4.89418857777999982694439040642465e-01 3.62188156504300029503440327971475e-01 2.47725502018499993406308590238041e-02
            3.59806957154999995651678545982577e-01 1.45288086625299989185222671039810e-01 4.94904956219699987407523167348700e-01 2.50209630570500003721168269521513e-02
            4.94144105509499997896938339181361e-01 3.61021638381800014094835660216631e-01 1.44834256108700043519377231859835e-01 2.52897293557499992788795140086222e-02
            1.44063068798099991330019520319183e-01 3.51350834188700000826344194138073e-01 5.04586097013200118865938748058397e-01 2.60018605093999995159226301666422e-02
            5.01976444000399979472604172769934e-01 1.43549166329300009747882427291188e-01 3.54474389670299983023937784309965e-01 2.60766783943000013401558589976048e-02
            3.55542383429799990768316320099984e-01 5.01649159950199985402718994009774e-01 1.42808456620000079340115917148069e-01 2.62449576179000013120834466917586e-02
            2.44343954077099995148003586109553e-01 2.40605212910399990366983047351823e-01 5.15050833012500097751740213425364e-01 2.99579881258000002564045871622511e-02
            2.43706498934199988770643585667131e-01 5.10901727705500019460771454760106e-01 2.45391773360300047279736190830590e-01 2.99804998713000016297147709565252e-02
            5.12220080732100013776175728708040e-01 2.45273797354299993855875072767958e-01 2.42506121913599992367949198524002e-01 2.99957636064499995331988912994348e-02
            2.52603831517800025974196387323900e-01 3.70031955509399990145169567767880e-01 3.77364212972799928369482813650393e-01 3.17066591724499968751871392669273e-02
            3.75989565285100024727427125981194e-01 2.50540661163099975805579333609785e-01 3.73469773551800054978144771666848e-01 3.17655930553999987786539804801578e-02
            3.72907798714399996153900929130032e-01 3.75375027754900003618132586780121e-01 2.51717173530699944716815252832021e-01 3.18603302835999968589675290786545e-02
            ];
        
        
    otherwise
        xyw=[];
end




pointset_stats=[
    13   36 3.94e-13    0    0
    16   55 7.82e-13    0    0
    18   66 5.01e-13    0    0
    20   78 7.68e-13    0    0
    ];


