
function [xyw_bar,pointset_stats]=set_vioreanu_rokhlin_C_barycentric(deg)

%--------------------------------------------------------------------
% CUBATURE RULE ON THE SIMPLEX [0 0; 0 1; 1 0] WITH ADE EQUAL TO deg.
% THE CUBATURE RULE IS DESCRIBED IN BARYCENTRICAL FORM.
%--------------------------------------------------------------------
% INPUT:
% deg: algebraic degree of precision (ADE) of the rule.
%--------------------------------------------------------------------
% OUTPUT:
% xyw_bar: MATRIX OF NODES xyw(:,1:3) IN BARYCENTRICAL COOR. AND WEIGHTS xyw(:,4).
%      REMARK: The sum of the weights makes 1.
%
% pointset_stats:
%    statistics of the pointset stored in a matrix. Each rows contains
%    degree, cardinality of the full pointset, moment error w.r.t.
%    orthn. Dubiner basis, number of points with negative weights,
%    number of points outside the domain.
%--------------------------------------------------------------------
% SOURCE:
%  B. Vioreanu and V. Rokhlin,
%  "Spectra of Multiplication Operators as a Numerical Tool",
%  Methods and Algorithms for Scientific Computing,
%  SIAM J. Sci. Comput., 36,1 (2014), pp.267-288.
%--------------------------------------------------------------------
% | DEG | CARD | W- | PT.O|   M.E.   |Q.|
%--------------------------------------------------------------------
% |   1 |   1 |   0 |   0 | 1.92e-16 |PI|
% |   2 |   3 |   0 |   0 | 1.42e-16 |PI|
% |   4 |   6 |   0 |   0 | 1.53e-16 |PI|
% |   5 |  10 |   0 |   0 | 3.05e-16 |PI|
% |   7 |  15 |   0 |   0 | 3.61e-16 |PI|
% |   8 |  21 |   0 |   0 | 5.62e-16 |PI|
% |  10 |  28 |   0 |   0 | 4.72e-16 |PI|
% |  12 |  36 |   0 |   0 | 4.44e-16 |PI|
% |  14 |  45 |   0 |   0 | 5.00e-16 |PI|
% |  15 |  55 |   0 |   0 | 4.44e-16 |PI|
% |  17 |  66 |   0 |   0 | 5.34e-16 |PI|
% |  19 |  78 |   0 |   0 | 8.53e-16 |PI|
% |  20 |  91 |   0 |   0 | 4.30e-16 |PI|
% |  22 | 105 |   0 |   0 | 5.69e-16 |PI|
% |  24 | 120 |   0 |   0 | 6.18e-16 |PI|
% |  25 | 136 |   0 |   0 | 6.38e-16 |PI|
% |  27 | 153 |   0 |   0 | 5.00e-16 |PI|
% |  28 | 171 |   0 |   0 | 5.27e-16 |PI|
% |  30 | 190 |   0 |   0 | 6.42e-16 |PI|
% |  32 | 210 |   0 |   0 | 6.83e-16 |PI|
%--------------------------------------------------------------------
% LEGEND:
%--------------------------------------------------------------------
% DEG  : algebraic degree of precision
% CARD : cardinality of the pointset
% W-   : negative weights
% PT.O : points outside the simplex [0 0; 1 0; 0 1].
% M.E. : maximum moment error in inf norm.
% Q.: quality of the rule.
%        PI: pos. weights, points in simplex
%        NI: some neg. weights, points in simplex
%        P0: pos. weights, some points out of the simplex
%        N0: some neg. weights, some points out of the simplex
%--------------------------------------------------------------------

switch deg
    
    case   1
        
        % ALG. DEG.:    1
        % PTS CARD.:    1
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.9e-16
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333425851918718763045e-01 1.00000000000000000000000000000000e+00
            ];
        
    case   2
        
        % ALG. DEG.:    2
        % PTS CARD.:    3
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.4e-16
        
        xyw_bar=[
            1.66666666666666601903656896865868e-01 1.66666666666666601903656896865868e-01 6.66666666666666740681534975010436e-01 3.33333333333333259318465024989564e-01
            1.66666666666666796192686206268263e-01 6.66666666666666518636930049979128e-01 1.66666666666666740681534975010436e-01 3.33333333333333536874221181278699e-01
            6.66666666666666740681534975010436e-01 1.66666666666666490881354434350214e-01 1.66666666666666768437110590639350e-01 3.33333333333333314829616256247391e-01
            ];
        
    case   4
        
        % ALG. DEG.:    4
        % PTS CARD.:    6
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.5e-16
        
        xyw_bar=[
            9.15762135097707430375635340169538e-02 9.15762135097707430375635340169538e-02 8.16847572980458513924872931966092e-01 1.09951743655321856896200927167229e-01
            9.15762135097707430375635340169538e-02 8.16847572980458513924872931966092e-01 9.15762135097707430375635340169538e-02 1.09951743655321856896200927167229e-01
            8.16847572980458513924872931966092e-01 9.15762135097707430375635340169538e-02 9.15762135097707430375635340169538e-02 1.09951743655321856896200927167229e-01
            4.45948490915964890213274429697776e-01 4.45948490915964890213274429697776e-01 1.08103018168070219573451140604448e-01 2.23381589678011444055627521265706e-01
            4.45948490915964890213274429697776e-01 1.08103018168070219573451140604448e-01 4.45948490915964890213274429697776e-01 2.23381589678011444055627521265706e-01
            1.08103018168070219573451140604448e-01 4.45948490915964890213274429697776e-01 4.45948490915964890213274429697776e-01 2.23381589678011444055627521265706e-01
            ];
        
    case   5
        
        % ALG. DEG.:    5
        % PTS CARD.:   10
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.1e-16
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333425851918718763045e-01 1.98965370459636470279463082988514e-01
            7.11129945334409846458001425162365e-02 7.11129945334410956681026050318906e-02 8.57774010933117891930521636822959e-01 6.02789804071406018493028966531710e-02
            7.11129945334409985235879503306933e-02 8.57774010933118002952824099338613e-01 7.11129945334409985235879503306933e-02 6.02789804071405602159394732098008e-02
            8.57774010933118002952824099338613e-01 7.11129945334409291346489112584095e-02 7.11129945334410679125269894029771e-02 6.02789804071405255214699536736589e-02
            6.14926166643125493571631068334682e-01 3.13979397676981020204323158395709e-01 7.10944356798934862240457732696086e-02 1.03366281386490302396374829640990e-01
            6.14926166643125493571631068334682e-01 7.10944356798936805130750826720032e-02 3.13979397676980798159718233364401e-01 1.03366281386490399540889484342188e-01
            3.13979397676980853670869464622228e-01 6.14926166643125493571631068334682e-01 7.10944356798935972463482357852627e-02 1.03366281386490399540889484342188e-01
            3.13979397676981020204323158395709e-01 7.10944356798934445906823498262384e-02 6.14926166643125493571631068334682e-01 1.03366281386490260763011406197620e-01
            7.10944356798934307128945420117816e-02 6.14926166643125493571631068334682e-01 3.13979397676981020204323158395709e-01 1.03366281386490205251860174939793e-01
            7.10944356798936805130750826720032e-02 3.13979397676981020204323158395709e-01 6.14926166643125271527026143303374e-01 1.03366281386490330151950445269904e-01
            ];
        
    case   7
        
        % ALG. DEG.:    7
        % PTS CARD.:   15
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.6e-16
        
        xyw_bar=[
            5.12530798414518040662102293936186e-02 5.12530798414517624328468059502484e-02 8.97493840317096447378730772470590e-01 3.20133347298796611113402832415886e-02
            5.12530798414517624328468059502484e-02 8.97493840317096447378730772470590e-01 5.12530798414517763106346137647051e-02 3.20133347298796194779768597982184e-02
            8.97493840317096447378730772470590e-01 5.12530798414518040662102293936186e-02 5.12530798414517485550589981357916e-02 3.20133347298796472335524754271319e-02
            4.74110794051135264748353392860736e-01 4.74110794051135320259504624118563e-01 5.17784118977293594809907517628744e-02 6.48798375470517746377296930404555e-02
            4.74110794051135209237202161602909e-01 5.17784118977294149921419830207014e-02 4.74110794051135375770655855376390e-01 6.48798375470517607599418852259987e-02
            5.17784118977294566255054064640717e-02 4.74110794051135320259504624118563e-01 4.74110794051135209237202161602909e-01 6.48798375470517746377296930404555e-02
            2.42184756344485996315540887735551e-01 2.42184756344485996315540887735551e-01 5.15630487311028007368918224528898e-01 1.27008925694331142874915485663223e-01
            2.42184756344486107337843350251205e-01 5.15630487311027785324313299497589e-01 2.42184756344486107337843350251205e-01 1.27008925694331198386066716921050e-01
            5.15630487311027785324313299497589e-01 2.42184756344486024071116503364465e-01 2.42184756344486190604570197137946e-01 1.27008925694331226141642332549964e-01
            7.18483300897606480539536732976558e-01 2.33346892984868881093518666602904e-01 4.81698061175246383669446004205383e-02 5.47156176810353958583910127799754e-02
            7.18483300897606480539536732976558e-01 4.81698061175246383669446004205383e-02 2.33346892984868881093518666602904e-01 5.47156176810353681028153971510619e-02
            2.33346892984868853337943050973990e-01 7.18483300897606480539536732976558e-01 4.81698061175246383669446004205383e-02 5.47156176810353750417093010582903e-02
            2.33346892984868908849094282231817e-01 4.81698061175245759168994652554829e-02 7.18483300897606591561839195492212e-01 5.47156176810353056527702619860065e-02
            4.81698061175246730614141199566802e-02 7.18483300897606480539536732976558e-01 2.33346892984868881093518666602904e-01 5.47156176810353750417093010582903e-02
            4.81698061175246106113689847916248e-02 2.33346892984868908849094282231817e-01 7.18483300897606480539536732976558e-01 5.47156176810353334083458776149200e-02
            ];
        
    case   8
        
        % ALG. DEG.:    8
        % PTS CARD.:   21
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.6e-16
        
        xyw_bar=[
            3.14057488009515869964438650185912e-02 3.14057488009516008742316728330479e-02 9.37188502398096812129324462148361e-01 1.37145072550597952459883188680578e-02
            3.14057488009515453630804415752209e-02 9.37188502398096923151626924664015e-01 3.14057488009514829130353064101655e-02 1.37145072550597640209657512855301e-02
            9.37188502398096923151626924664015e-01 3.14057488009515384241865376679925e-02 3.14057488009515384241865376679925e-02 1.37145072550597709598596551927585e-02
            1.96019357807156596740227882946783e-01 1.96019357807156568984652267317870e-01 6.07961284385686751008393002848607e-01 7.43976090198317785429082960035885e-02
            1.96019357807156707762530345462437e-01 6.07961284385686751008393002848607e-01 1.96019357807156513473501036060043e-01 7.43976090198318340540595272614155e-02
            6.07961284385686751008393002848607e-01 1.96019357807156596740227882946783e-01 1.96019357807156652251379114204610e-01 7.43976090198317924206961038180452e-02
            4.04461999739164801415114425253705e-01 4.04461999739164801415114425253705e-01 1.91076000521670452680922380750417e-01 8.50145322145201554064897209173068e-02
            4.04461999739164801415114425253705e-01 1.91076000521670452680922380750417e-01 4.04461999739164801415114425253705e-01 8.50145322145201970398531443606771e-02
            1.91076000521670313903044302605849e-01 4.04461999739164801415114425253705e-01 4.04461999739164912437416887769359e-01 8.50145322145201831620653365462204e-02
            7.87642410206076615253323325305246e-01 1.71130424662843866334682729757333e-01 4.12271651310795184119939449374215e-02 3.84058450381026955655627830310550e-02
            7.87642410206076615253323325305246e-01 4.12271651310795253508878488446499e-02 1.71130424662843866334682729757333e-01 3.84058450381027233211383986599685e-02
            1.71130424662843894090258345386246e-01 7.87642410206076615253323325305246e-01 4.12271651310794906564183293085080e-02 3.84058450381026955655627830310550e-02
            1.71130424662843894090258345386246e-01 4.12271651310794767786305214940512e-02 7.87642410206076615253323325305246e-01 3.84058450381026886266688791238266e-02
            4.12271651310795322897817527518782e-02 7.87642410206076615253323325305246e-01 1.71130424662843894090258345386246e-01 3.84058450381027233211383986599685e-02
            4.12271651310795808620390801024769e-02 1.71130424662843894090258345386246e-01 7.87642410206076504231020862789592e-01 3.84058450381027025044566869382834e-02
            5.84632756543619902522834763658466e-01 3.76058771577794492824153849142021e-01 3.93084718785856046530113871995127e-02 4.16974973838580634244443956504256e-02
            5.84632756543619902522834763658466e-01 3.93084718785856948586321379934816e-02 3.76058771577794381801851386626367e-01 4.16974973838580356688687800215121e-02
            3.76058771577794381801851386626367e-01 5.84632756543619902522834763658466e-01 3.93084718785857711864650809729937e-02 4.16974973838580009743992604853702e-02
            3.76058771577794437313002617884194e-01 3.93084718785857503697833692513086e-02 5.84632756543619791500532301142812e-01 4.16974973838580426077626839287404e-02
            3.93084718785857087364199458079383e-02 5.84632756543619902522834763658466e-01 3.76058771577794437313002617884194e-01 4.16974973838580634244443956504256e-02
            3.93084718785856115919052911067411e-02 3.76058771577794381801851386626367e-01 5.84632756543620013545137226174120e-01 4.16974973838579801577175487636850e-02
            ];
        
    case  10
        
        % ALG. DEG.:   10
        % PTS CARD.:   28
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.7e-16
        
        xyw_bar=[
            3.33333333333333536874221181278699e-01 3.33333333333333314829616256247391e-01 3.33333333333333203807313793731737e-01 8.08162600508437051205490320171521e-02
            2.27066403128517654841367345852632e-02 2.27066403128517411980080709099639e-02 9.54586719374296510665089954272844e-01 7.38163289619725090928703536974353e-03
            2.27066403128517481369019748171922e-02 9.54586719374296510665089954272844e-01 2.27066403128517446674550228635781e-02 7.38163289619725611345746330016482e-03
            9.54586719374296621687392416788498e-01 2.27066403128517065035385513738220e-02 2.27066403128516718090690318376801e-02 7.38163289619723442941401359007614e-03
            4.85348136374768979184324280140572e-01 4.85348136374769034695475511398399e-01 2.93037272504620416313514397188555e-02 2.64788860014971601242184107150024e-02
            4.85348136374768979184324280140572e-01 2.93037272504619444868367850176583e-02 4.85348136374769145717777973914053e-01 2.64788860014971358380897470397031e-02
            2.93037272504620277535636319043988e-02 4.85348136374768979184324280140572e-01 4.85348136374768979184324280140572e-01 2.64788860014971774714531704830733e-02
            1.55015517336222874433815377415158e-01 1.55015517336222957700542224301898e-01 6.89968965327554140110066782654030e-01 5.01093142308751057778515303198219e-02
            1.55015517336222985456117839930812e-01 6.89968965327554029087764320138376e-01 1.55015517336222985456117839930812e-01 5.01093142308751265945332420415070e-02
            6.89968965327554029087764320138376e-01 1.55015517336222985456117839930812e-01 1.55015517336222985456117839930812e-01 5.01093142308751196556393381342787e-02
            8.38551707380394306667881210159976e-01 1.30053450518161112237081056264287e-01 3.13948421014445810950377335757366e-02 2.35021042128849694252679114470084e-02
            8.38551707380394306667881210159976e-01 3.13948421014445602783560218540515e-02 1.30053450518161139992656671893201e-01 2.35021042128849763641618153542368e-02
            1.30053450518161306526110365666682e-01 8.38551707380394084623276285128668e-01 3.13948421014446088506133492046501e-02 2.35021042128849728947148634006226e-02
            1.30053450518161251014959134408855e-01 3.13948421014445255838865023179096e-02 8.38551707380394306667881210159976e-01 2.35021042128849624863740075397800e-02
            3.13948421014445602783560218540515e-02 8.38551707380394195645578747644322e-01 1.30053450518161195503807903151028e-01 2.35021042128849867725026712150793e-02
            3.13948421014445255838865023179096e-02 1.30053450518161195503807903151028e-01 8.38551707380394306667881210159976e-01 2.35021042128849451391392477717091e-02
            6.66920601059608042504578406806104e-01 3.02790097367536925609954323590500e-01 3.02893015728550318854672696033958e-02 2.85261227456073407360293003875995e-02
            6.66920601059607931482275944290450e-01 3.02893015728550631104898371859235e-02 3.02790097367536981121105554848327e-01 2.85261227456073511443701562484421e-02
            3.02790097367536981121105554848327e-01 6.66920601059607931482275944290450e-01 3.02893015728550318854672696033958e-02 2.85261227456073303276884445267569e-02
            3.02790097367536981121105554848327e-01 3.02893015728550943355124047684512e-02 6.66920601059607820459973481774796e-01 2.85261227456073546138171082020563e-02
            3.02893015728550631104898371859235e-02 6.66920601059607931482275944290450e-01 3.02790097367537036632256786106154e-01 2.85261227456073580832640601556704e-02
            3.02893015728551116827471645365222e-02 3.02790097367536925609954323590500e-01 6.66920601059607931482275944290450e-01 2.85261227456073511443701562484421e-02
            5.20560306058803101691978554299567e-01 3.30714245940923079469797585261404e-01 1.48725448000273818838223860439030e-01 5.91841464687489424156829898038268e-02
            5.20560306058802990669676091783913e-01 1.48725448000273791082648244810116e-01 3.30714245940923246003251279034885e-01 5.91841464687489771101525093399687e-02
            3.30714245940923190492100047777058e-01 5.20560306058802990669676091783913e-01 1.48725448000273874349375091696857e-01 5.91841464687489632323647015255119e-02
            3.30714245940923079469797585261404e-01 1.48725448000273846593799476067943e-01 5.20560306058802990669676091783913e-01 5.91841464687489771101525093399687e-02
            1.48725448000273874349375091696857e-01 5.20560306058802990669676091783913e-01 3.30714245940923134980948816519231e-01 5.91841464687489771101525093399687e-02
            1.48725448000273929860526322954684e-01 3.30714245940923079469797585261404e-01 5.20560306058802879647373629268259e-01 5.91841464687490187435159327833389e-02
            ];
        
    case  12
        
        % ALG. DEG.:   12
        % PTS CARD.:   36
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.4e-16
        
        xyw_bar=[
            1.93652109024277166438032793394086e-02 1.93652109024277131743563273857944e-02 9.61269578195144580590181249135640e-01 5.14135021968479752341174560115178e-03
            1.93652109024277270521441352002512e-02 9.61269578195144580590181249135640e-01 1.93652109024277097049093754321802e-02 5.14135021968478277826219979829148e-03
            9.61269578195144580590181249135640e-01 1.93652109024277339910380391074796e-02 1.93652109024276854187807117568809e-02 5.14135021968478711507088974030921e-03
            1.20430831569190488483656054086168e-01 1.20430831569190682772685363488563e-01 7.59138336861618912010385429312009e-01 3.16396426284812973261928448209801e-02
            1.20430831569190530117019477529539e-01 7.59138336861618912010385429312009e-01 1.20430831569190543994807285343995e-01 3.16396426284812418150416135631531e-02
            7.59138336861618912010385429312009e-01 1.20430831569190530117019477529539e-01 1.20430831569190557872595093158452e-01 3.16396426284812209983599018414679e-02
            4.40005690210435673748179397080094e-01 4.40005690210435562725876934564440e-01 1.19988619579128763525943668355467e-01 4.67416591710063458742219211217162e-02
            4.40005690210435562725876934564440e-01 1.19988619579128805159307091798837e-01 4.40005690210435618237028165822267e-01 4.67416591710063597520097289361729e-02
            1.19988619579128763525943668355467e-01 4.40005690210435840281633090853575e-01 4.40005690210435396192423240790959e-01 4.67416591710063042408584976783459e-02
            2.71524632000497700978769444191130e-01 2.71524632000497700978769444191130e-01 4.56950735999004598042461111617740e-01 6.24578255383045066739811090883450e-02
            2.71524632000497700978769444191130e-01 4.56950735999004598042461111617740e-01 2.71524632000497700978769444191130e-01 6.24578255383045274906628208100301e-02
            4.56950735999004653553612342875567e-01 2.71524632000497645467618212933303e-01 2.71524632000497756489920675448957e-01 6.24578255383045066739811090883450e-02
            8.69373866882279466850036442338023e-01 1.06407946399805108939062847639434e-01 2.42181867179154242109007100225426e-02 1.48143538290032034804299954089402e-02
            8.69373866882279466850036442338023e-01 2.42181867179154276803476619761568e-02 1.06407946399805108939062847639434e-01 1.48143538290032069498769473625543e-02
            1.06407946399805122816850655453891e-01 8.69373866882279466850036442338023e-01 2.42181867179154242109007100225426e-02 1.48143538290032000109830434553260e-02
            1.06407946399805136694638463268348e-01 2.42181867179154103331129022080859e-02 8.69373866882279466850036442338023e-01 1.48143538290032138887708512697827e-02
            2.42181867179154033942189983008575e-02 8.69373866882279577872338904853677e-01 1.06407946399804997916760385123780e-01 1.48143538290031896026421875944834e-02
            2.42181867179154207414537580689284e-02 1.06407946399805108939062847639434e-01 8.69373866882279466850036442338023e-01 1.48143538290031878679187116176763e-02
            5.62233442343925027273598971078172e-01 4.14262053470467184190795251197414e-01 2.35045041856077885356057777244132e-02 1.83093224679997459980462792827893e-02
            5.62233442343925027273598971078172e-01 2.35045041856077469022423542810429e-02 4.14262053470467239701946482455241e-01 1.83093224679997390591523753755610e-02
            4.14262053470467184190795251197414e-01 5.62233442343925027273598971078172e-01 2.35045041856077885356057777244132e-02 1.83093224679997355897054234219468e-02
            4.14262053470467184190795251197414e-01 2.35045041856077538411362581882713e-02 5.62233442343925027273598971078172e-01 1.83093224679997251813645675611042e-02
            2.35045041856077399633484503738146e-02 5.62233442343925027273598971078172e-01 4.14262053470467184190795251197414e-01 1.83093224679997459980462792827893e-02
            2.35045041856077295550075945129720e-02 4.14262053470467184190795251197414e-01 5.62233442343925138295901433593826e-01 1.83093224679997182424706636538758e-02
            7.25103351867971257505018911615480e-01 2.52014086602536824255338387956726e-01 2.28825615294919182396427004277939e-02 1.90258494711416613343057946394765e-02
            7.25103351867971257505018911615480e-01 2.28825615294920119147104031753770e-02 2.52014086602536713233035925441072e-01 1.90258494711416613343057946394765e-02
            2.52014086602536824255338387956726e-01 7.25103351867971146482716449099826e-01 2.28825615294919737507939316856209e-02 1.90258494711416543954118907322481e-02
            2.52014086602536824255338387956726e-01 2.28825615294920257924982109898338e-02 7.25103351867971146482716449099826e-01 1.90258494711416960287753141756184e-02
            2.28825615294919980369225953609202e-02 7.25103351867971257505018911615480e-01 2.52014086602536768744187156698899e-01 1.90258494711416890898814102683900e-02
            2.28825615294919876285817395000777e-02 2.52014086602536824255338387956726e-01 7.25103351867971257505018911615480e-01 1.90258494711416717426466505003191e-02
            6.17412891519986017918597553943982e-01 2.66109864468771051271289707074175e-01 1.16477244011242930810112738981843e-01 4.15269021197835763725692004300072e-02
            6.17412891519986017918597553943982e-01 1.16477244011242916932324931167386e-01 2.66109864468771051271289707074175e-01 4.15269021197835763725692004300072e-02
            2.66109864468771106782440938332002e-01 6.17412891519985906896295091428328e-01 1.16477244011242930810112738981843e-01 4.15269021197835902503570082444639e-02
            2.66109864468771106782440938332002e-01 1.16477244011242972443476162425213e-01 6.17412891519985906896295091428328e-01 4.15269021197835624947813926155504e-02
            1.16477244011242972443476162425213e-01 6.17412891519986017918597553943982e-01 2.66109864468771051271289707074175e-01 4.15269021197835486169935848010937e-02
            1.16477244011242930810112738981843e-01 2.66109864468771384338197094621137e-01 6.17412891519985684851690166397020e-01 4.15269021197835833114631043372356e-02
            ];
        
    case  14
        
        % ALG. DEG.:   14
        % PTS CARD.:   45
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.0e-16
        
        xyw_bar=[
            1.88250991695916025658874559667311e-02 1.88250991695915990964405040131169e-02 9.62349801660816739357073856808711e-01 4.62291117115323172553909003568151e-03
            1.88250991695915852186526961986601e-02 9.62349801660816739357073856808711e-01 1.88250991695916303214630715956446e-02 4.62291117115322999081561405887442e-03
            9.62349801660816739357073856808711e-01 1.88250991695915990964405040131169e-02 1.88250991695916615464856391781723e-02 4.62291117115323346026256601248861e-03
            4.90249109938372285011354279049556e-01 4.90249109938372229500203047791729e-01 1.95017801232554299772914419008885e-02 1.43428868364703244325797371061526e-02
            4.90249109938372285011354279049556e-01 1.95017801232554889578896251123297e-02 4.90249109938372173989051816533902e-01 1.43428868364703365756440689438023e-02
            1.95017801232555028356774329267864e-02 4.90249109938372285011354279049556e-01 4.90249109938372173989051816533902e-01 1.43428868364703539228788287118732e-02
            8.19725416516456917292998696211725e-02 8.19725416516453586623924820742104e-02 8.36054916696708949608307648304617e-01 1.73982229394614024142740049683198e-02
            8.19725416516457056070876774356293e-02 8.36054916696708616541400260757655e-01 8.19725416516456917292998696211725e-02 1.73982229394614440476374284116901e-02
            8.36054916696708616541400260757655e-01 8.19725416516456084625730227344320e-02 8.19725416516457749960267165079131e-02 1.73982229394614405781904764580759e-02
            2.20530835565724586677660568057036e-01 2.20530835565724503410933721170295e-01 5.58938328868550882155830095143756e-01 4.53860877671218160278598929835425e-02
            2.20530835565724447899782489912468e-01 5.58938328868550771133527632628102e-01 2.20530835565724725455538646201603e-01 4.53860877671218160278598929835425e-02
            5.58938328868550549088922707596794e-01 2.20530835565724753211114261830517e-01 2.20530835565724697699963030572690e-01 4.53860877671218299056477007979993e-02
            3.86124189278291174431956278567668e-01 3.86124189278291507498863666114630e-01 2.27751621443417318069180055317702e-01 5.21922928761801019348354202520568e-02
            3.86124189278291285454258741083322e-01 2.27751621443417290313604439688788e-01 3.86124189278291396476561203598976e-01 5.21922928761801019348354202520568e-02
            2.27751621443417318069180055317702e-01 3.86124189278291229943107509825495e-01 3.86124189278291451987712434856803e-01 5.21922928761801019348354202520568e-02
            8.87822775364344174420239141909406e-01 9.67912764708031281291056302507059e-02 1.53859481648526974506552278398885e-02 8.20408190048361671276033746380563e-03
            8.87822775364344174420239141909406e-01 1.53859481648527408187421272600659e-02 9.67912764708030864957422068073356e-02 8.20408190048360977386643355657725e-03
            9.67912764708029893512275521061383e-02 8.87822775364344285442541604425060e-01 1.53859481648527252062308434688020e-02 8.20408190048361497803686148699853e-03
            9.67912764708030587401665911784221e-02 1.53859481648526350006100926748331e-02 8.87822775364344285442541604425060e-01 8.20408190048358028356734195085664e-03
            1.53859481648527599007003630049439e-02 8.87822775364344063397936679393752e-01 9.67912764708032113958324771374464e-02 8.20408190048362885582466930145529e-03
            1.53859481648527425534656032368730e-02 9.67912764708030864957422068073356e-02 8.87822775364344174420239141909406e-01 8.20408190048362018220728941741982e-03
            7.67286163241992502648258778208401e-01 2.14478157320871859470301501460199e-01 1.82356794371356378814397203314002e-02 1.18935919339511631670491809131818e-02
            7.67286163241992502648258778208401e-01 1.82356794371357107398257113572981e-02 2.14478157320871776203574654573458e-01 1.18935919339512013309656524029378e-02
            2.14478157320871803959150270202372e-01 7.67286163241992502648258778208401e-01 1.82356794371356656370153359603137e-02 1.18935919339512013309656524029378e-02
            2.14478157320871803959150270202372e-01 1.82356794371356413508866722850144e-02 7.67286163241992502648258778208401e-01 1.18935919339511649017726568899889e-02
            1.82356794371356274730988644705576e-02 7.67286163241992502648258778208401e-01 2.14478157320871831714725885831285e-01 1.18935919339511302073031373538470e-02
            1.82356794371356482897805761922427e-02 2.14478157320871803959150270202372e-01 7.67286163241992502648258778208401e-01 1.18935919339511666364961328667960e-02
            6.33349939282431662057604171423009e-01 3.48145146811778050111030324842432e-01 1.85049139057902878313655037345598e-02 1.31258507832268285497878679279893e-02
            6.33349939282431551035301708907355e-01 1.85049139057902913008124556881739e-02 3.48145146811778161133332787358086e-01 1.31258507832268372234052478120248e-02
            3.48145146811778105622181556100259e-01 6.33349939282431551035301708907355e-01 1.85049139057903433425167349923868e-02 1.31258507832268493664695796496744e-02
            3.48145146811778105622181556100259e-01 1.85049139057902843619185517809456e-02 6.33349939282431662057604171423009e-01 1.31258507832268337539582958584106e-02
            1.85049139057903155869411193634733e-02 6.33349939282431773079906633938663e-01 3.48145146811777883577576631068951e-01 1.31258507832268337539582958584106e-02
            1.85049139057903294647289271779300e-02 3.48145146811777994599879093584605e-01 6.33349939282431662057604171423009e-01 1.31258507832268406928521997656389e-02
            7.07553018466495098337531999277417e-01 2.00642015647370303144114700444334e-01 9.18049658861345985183533002782497e-02 2.89312304714396846538893015576832e-02
            7.07553018466495098337531999277417e-01 9.18049658861347234184435706083605e-02 2.00642015647370164366236622299766e-01 2.89312304714396568983136859287697e-02
            2.00642015647370219877387853557593e-01 7.07553018466495098337531999277417e-01 9.18049658861346262739289159071632e-02 2.89312304714396499594197820215413e-02
            2.00642015647370303144114700444334e-01 9.18049658861345291294142612059659e-02 7.07553018466495209359834461793071e-01 2.89312304714396395510789261606988e-02
            9.18049658861345013738386455770524e-02 7.07553018466495098337531999277417e-01 2.00642015647370386410841547331074e-01 2.89312304714396707761014937432265e-02
            9.18049658861345985183533002782497e-02 2.00642015647369859054904850381718e-01 7.07553018466495542426741849340033e-01 2.89312304714396464899728300679271e-02
            5.41383214828122127926235407358035e-01 3.61345725637031767174534024888999e-01 9.72710595348461048992305677529657e-02 3.75407107823720123662702974343119e-02
            5.41383214828122349970840332389344e-01 9.72710595348459799991402974228549e-02 3.61345725637031656152231562373345e-01 3.75407107823719984884824896198552e-02
            3.61345725637031489618777868599864e-01 5.41383214828122460993142794904998e-01 9.72710595348460493880793364951387e-02 3.75407107823719915495885857126268e-02
            3.61345725637031711663382793631172e-01 9.72710595348459522435646817939414e-02 5.41383214828122349970840332389344e-01 3.75407107823719776718007778981701e-02
            9.72710595348459661213524896083982e-02 5.41383214828122127926235407358035e-01 3.61345725637031933707987718662480e-01 3.75407107823720123662702974343119e-02
            9.72710595348459938769281052373117e-02 3.61345725637031323085324174826383e-01 5.41383214828122572015445257420652e-01 3.75407107823719846106946818053984e-02
            ];
        
    case  15
        
        % ALG. DEG.:   15
        % PTS CARD.:   55
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.4e-16
        
        xyw_bar=[
            3.33333333333333370340767487505218e-01 3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 4.47752981925974707921866979631886e-02
            1.76519929574458882270970860872694e-02 1.76519929574459194521196536697971e-02 9.64696014085108188851336308289319e-01 3.91210040603462341074481045666289e-03
            1.76519929574459229215666056234113e-02 9.64696014085108188851336308289319e-01 1.76519929574459055743318458553404e-02 3.91210040603463728853261827111965e-03
            9.64696014085108188851336308289319e-01 1.76519929574459298604605095306397e-02 1.76519929574458812882031821800410e-02 3.91210040603463208436219034069836e-03
            7.71703842193710437546982916501292e-02 7.71703842193710021213348682067590e-02 8.45659231561257884735027801070828e-01 1.65337256371640463348260396969636e-02
            7.71703842193708772212445978766482e-02 8.45659231561258217801935188617790e-01 7.71703842193709466101836369489320e-02 1.65337256371640289875912799288926e-02
            8.45659231561258106779632726102136e-01 7.71703842193709327323958291344752e-02 7.71703842193709604879714447633887e-02 1.65337256371640324570382318825068e-02
            4.59371797617381893985566421179101e-01 4.59371797617381893985566421179101e-01 8.12564047652362120288671576417983e-02 2.06164703556393699712678824198520e-02
            4.59371797617381893985566421179101e-01 8.12564047652361287621403107550577e-02 4.59371797617382005007868883694755e-01 2.06164703556393005823288433475682e-02
            8.12564047652361703955037341984280e-02 4.59371797617381893985566421179101e-01 4.59371797617381893985566421179101e-01 2.06164703556393213990105550692533e-02
            2.04922897588778768485084924577677e-01 2.04922897588778796240660540206591e-01 5.90154204822442407518678919586819e-01 3.19653987298611497180900187231600e-02
            2.04922897588778823996236155835504e-01 5.90154204822442407518678919586819e-01 2.04922897588778796240660540206591e-01 3.19653987298611635958778265376168e-02
            5.90154204822442407518678919586819e-01 2.04922897588778851751811771464418e-01 2.04922897588778740729509308948764e-01 3.19653987298611566569839226303884e-02
            9.03429699782028916565934650861891e-01 8.35136443388275473420634398280527e-02 1.30566558791435360920019093100564e-02 5.67392893537029506523250432792338e-03
            9.03429699782028805543632188346237e-01 1.30566558791435395614488612636706e-02 8.35136443388276583643659023437067e-02 5.67392893537030894302031214238013e-03
            8.35136443388276306087902867147932e-02 9.03429699782028805543632188346237e-01 1.30566558791436193587287561967969e-02 5.67392893537030113676467024674821e-03
            8.35136443388274779531244007557689e-02 1.30566558791435829295357606838479e-02 9.03429699782028916565934650861891e-01 5.67392893537031674927595403801206e-03
            1.30566558791435135405967216115641e-02 9.03429699782028916565934650861891e-01 8.35136443388275750976390554569662e-02 5.67392893537029506523250432792338e-03
            1.30566558791435829295357606838479e-02 8.35136443388275057087000163846824e-02 9.03429699782028916565934650861891e-01 5.67392893537031588191421604960851e-03
            8.10386575943458420390186347503914e-01 1.70743649147708009827795194723876e-01 1.88697749088335697820184577722102e-02 8.97907823151042165654711624256379e-03
            8.10386575943458420390186347503914e-01 1.88697749088335836598062655866670e-02 1.70743649147708009827795194723876e-01 8.97907823151042339127059221937088e-03
            1.70743649147708009827795194723876e-01 8.10386575943458420390186347503914e-01 1.88697749088335697820184577722102e-02 8.97907823151040777875930842810703e-03
            1.70743649147708009827795194723876e-01 1.88697749088335316181019862824542e-02 8.10386575943458420390186347503914e-01 8.97907823151043553433492405702054e-03
            1.88697749088335732514654097258244e-02 8.10386575943458420390186347503914e-01 1.70743649147708009827795194723876e-01 8.97907823151040951348278440491413e-03
            1.88697749088335385569958901896825e-02 1.70743649147708009827795194723876e-01 8.10386575943458420390186347503914e-01 8.97907823151040951348278440491413e-03
            7.02894663916247330881503785349196e-01 2.80682296607787540310141594090965e-01 1.64230394759651288083546205598395e-02 1.07165968553832070819797905869564e-02
            7.02894663916247330881503785349196e-01 1.64230394759651843195058518176666e-02 2.80682296607787484798990362833138e-01 1.07165968553832140208736944941847e-02
            2.80682296607787429287839131575311e-01 7.02894663916247441903806247864850e-01 1.64230394759651288083546205598395e-02 1.07165968553832088167032665637635e-02
            2.80682296607787651332444056606619e-01 1.64230394759652051361875635393517e-02 7.02894663916247108836898860317888e-01 1.07165968553832122861502185173777e-02
            1.64230394759651877889528037712807e-02 7.02894663916247330881503785349196e-01 2.80682296607787429287839131575311e-01 1.07165968553832174903206464477989e-02
            1.64230394759651981972936596321233e-02 2.80682296607787484798990362833138e-01 7.02894663916247219859201322833542e-01 1.07165968553832088167032665637635e-02
            5.64636464683940353381785826059058e-01 4.19465101758232095097156388874282e-01 1.58984335578275515210577850666596e-02 1.17235804303089351241506932410630e-02
            5.64636464683940353381785826059058e-01 1.58984335578275411127169292058170e-02 4.19465101758232095097156388874282e-01 1.17235804303089333894272172642559e-02
            4.19465101758231984074853926358628e-01 5.64636464683940464404088288574712e-01 1.58984335578274960099065538088325e-02 1.17235804303089403283211211714843e-02
            4.19465101758231984074853926358628e-01 1.58984335578275341738230252985886e-02 5.64636464683940464404088288574712e-01 1.17235804303089143074689815193778e-02
            1.58984335578275480516108331130454e-02 5.64636464683940353381785826059058e-01 4.19465101758232150608307620132109e-01 1.17235804303089420630445971482914e-02
            1.58984335578275133571413135769035e-02 4.19465101758231984074853926358628e-01 5.64636464683940575426390751090366e-01 1.17235804303089212463628854266062e-02
            5.84478584036716264371591478266055e-01 3.33042061006165956982272291497793e-01 8.24793549571177786461362302361522e-02 2.52243533019002909301153891874492e-02
            5.84478584036716264371591478266055e-01 8.24793549571176815016215755349549e-02 3.33042061006166068004574754013447e-01 2.52243533019003013384562450482917e-02
            3.33042061006165901471121060239966e-01 5.84478584036716375393893940781709e-01 8.24793549571176676238337677204981e-02 2.52243533019002909301153891874492e-02
            3.33042061006166123515725985271274e-01 8.24793549571177092571971911638684e-02 5.84478584036716153349289015750401e-01 2.52243533019002770523275813729924e-02
            8.24793549571176953794093833494117e-02 5.84478584036716264371591478266055e-01 3.33042061006166068004574754013447e-01 2.52243533019002978690092930946776e-02
            8.24793549571176953794093833494117e-02 3.33042061006166123515725985271274e-01 5.84478584036716153349289015750401e-01 2.52243533019002943995623411410634e-02
            7.17030852299470722321927951270482e-01 1.92513233859061272390533758880338e-01 9.04559138414680052875382898491807e-02 2.58199751262594839884556563447404e-02
            7.17030852299470611299625488754828e-01 9.04559138414681024320529445503780e-02 1.92513233859061272390533758880338e-01 2.58199751262594978662434641591972e-02
            1.92513233859061161368231296364684e-01 7.17030852299470722321927951270482e-01 9.04559138414681163098407523648348e-02 2.58199751262594909273495602519688e-02
            1.92513233859061355657260605767078e-01 9.04559138414680607986895211070077e-02 7.17030852299470611299625488754828e-01 2.58199751262595013356904161128114e-02
            9.04559138414681024320529445503780e-02 7.17030852299470611299625488754828e-01 1.92513233859061272390533758880338e-01 2.58199751262595048051373680664256e-02
            9.04559138414680746764773289214645e-02 1.92513233859061355657260605767078e-01 7.17030852299470500277323026239173e-01 2.58199751262595082745843200200397e-02
            4.72245408246903863869192718993872e-01 3.36118950295532514793706013733754e-01 1.91635641457563621337101267272374e-01 3.45527565228181468559576217103313e-02
            4.72245408246903919380343950251699e-01 1.91635641457563538070374420385633e-01 3.36118950295532625816008476249408e-01 3.45527565228181676726393334320164e-02
            3.36118950295532514793706013733754e-01 4.72245408246903919380343950251699e-01 1.91635641457563565825950036014547e-01 3.45527565228181329781698138958745e-02
            3.36118950295532514793706013733754e-01 1.91635641457563593581525651643460e-01 4.72245408246903863869192718993872e-01 3.45527565228181676726393334320164e-02
            1.91635641457563593581525651643460e-01 4.72245408246903919380343950251699e-01 3.36118950295532459282554782475927e-01 3.45527565228181537948515256175597e-02
            1.91635641457563621337101267272374e-01 3.36118950295532514793706013733754e-01 4.72245408246903863869192718993872e-01 3.45527565228181607337454295247881e-02
            ];
        
    case  17
        
        % ALG. DEG.:   17
        % PTS CARD.:   66
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.3e-16
        
        xyw_bar=[
            1.33804426737547284276264036861903e-02 1.33804426737547266929029277093832e-02 9.73239114652490577839216712163761e-01 2.26657506916947033837050895499488e-03
            1.33804426737546659775812685211349e-02 9.73239114652490688861519174679415e-01 1.33804426737546000580891814024653e-02 2.26657506916945862898704611154699e-03
            9.73239114652490577839216712163761e-01 1.33804426737547110803916439181194e-02 1.33804426737547110803916439181194e-02 2.26657506916945906266791510574876e-03
            4.92937508753781006287653099207091e-01 4.92937508753780784243048174175783e-01 1.41249824924381539581474953592988e-02 9.20461048313055661684867203575777e-03
            4.92937508753781006287653099207091e-01 1.41249824924381661012118271969484e-02 4.92937508753780784243048174175783e-01 9.20461048313055141267824410533649e-03
            1.41249824924381556928709713361059e-02 4.92937508753781006287653099207091e-01 4.92937508753780784243048174175783e-01 9.20461048313055141267824410533649e-03
            6.86030848984772734233672508707969e-02 6.86030848984772734233672508707969e-02 8.62793830203045342130963035742752e-01 1.20545682898302898566900154264658e-02
            6.86030848984773844456697133864509e-02 8.62793830203045342130963035742752e-01 6.86030848984773289345184821286239e-02 1.20545682898302915914134914032729e-02
            8.62793830203045342130963035742752e-01 6.86030848984773150567306743141671e-02 6.86030848984773428123062899430806e-02 1.20545682898303054692012992177297e-02
            1.63281281675213801873880470338918e-01 1.63281281675213940651758548483485e-01 6.73437436649572229718785365548683e-01 2.23442891612571359338801357807824e-02
            1.63281281675213718607153623452177e-01 6.73437436649572340741087828064337e-01 1.63281281675213940651758548483485e-01 2.23442891612571532811148955488534e-02
            6.73437436649572340741087828064337e-01 1.63281281675213718607153623452177e-01 1.63281281675213940651758548483485e-01 2.23442891612571359338801357807824e-02
            4.17510360602973917831803873923491e-01 4.17510360602973806809501411407837e-01 1.64979278794052219847543483410846e-01 3.12656695521544500393495980006264e-02
            4.17510360602973695787198948892183e-01 1.64979278794052386380997177184327e-01 4.17510360602973862320652642665664e-01 3.12656695521544292226678862789413e-02
            1.64979278794052275358694714668673e-01 4.17510360602973917831803873923491e-01 4.17510360602973806809501411407837e-01 3.12656695521544292226678862789413e-02
            2.86487437482899232055189031598275e-01 2.86487437482899343077491494113929e-01 4.27025125034201480378470705545624e-01 3.68126519077739758345479970103042e-02
            2.86487437482899343077491494113929e-01 4.27025125034201313845017011772143e-01 2.86487437482899287566340262856102e-01 3.68126519077739897123358048247610e-02
            4.27025125034201369356168243029970e-01 2.86487437482899232055189031598275e-01 2.86487437482899343077491494113929e-01 3.68126519077739827734419009175326e-02
            9.22032114610694231160437084326986e-01 6.50938297812631572192287876532646e-02 1.28740556080426116203341280197492e-02 4.45717468698359972623102009947615e-03
            9.22032114610694453205042009358294e-01 1.28740556080425300883307571098157e-02 6.50938297812630184413507095086970e-02 4.45717468698360493040144802989744e-03
            6.50938297812631294636531720243511e-02 9.22032114610694342182739546842640e-01 1.28740556080425561091828967619222e-02 4.45717468698360753248666199510808e-03
            6.50938297812632682415312501689186e-02 1.28740556080424867202438576896384e-02 9.22032114610694231160437084326986e-01 4.45717468698360926721013797191517e-03
            1.28740556080425474355655168778867e-02 9.22032114610694231160437084326986e-01 6.50938297812632127303800189110916e-02 4.45717468698361360401882791393291e-03
            1.28740556080424745771795258519887e-02 6.50938297812632404859556345400051e-02 9.22032114610694342182739546842640e-01 4.45717468698359712414580613426551e-03
            8.42560091151481893589902938401792e-01 1.43419095062200824264309062527900e-01 1.40208137863172821457879990703077e-02 6.31122674550474351690132479575368e-03
            8.42560091151481893589902938401792e-01 1.40208137863172769416175711398864e-02 1.43419095062200824264309062527900e-01 6.31122674550475045579522870298206e-03
            1.43419095062200907531035909414641e-01 8.42560091151481782567600475886138e-01 1.40208137863172543902123834413942e-02 6.31122674550475132315696669138561e-03
            1.43419095062200907531035909414641e-01 1.40208137863173289833218504440993e-02 8.42560091151481671545298013370484e-01 6.31122674550475045579522870298206e-03
            1.40208137863172873499584270007290e-02 8.42560091151481893589902938401792e-01 1.43419095062200852019884678156814e-01 6.31122674550472877175177899289338e-03
            1.40208137863173220444279465368709e-02 1.43419095062200907531035909414641e-01 8.42560091151481671545298013370484e-01 6.31122674550474525162480077256077e-03
            7.43449702189565297949513933417620e-01 2.43155281870591571946604858567298e-01 1.33950159398431301038812080150819e-02 7.64568774392988988186647958400499e-03
            7.43449702189565297949513933417620e-01 1.33950159398430919399647365253259e-02 2.43155281870591599702180474196211e-01 7.64568774392989595339864550282982e-03
            2.43155281870591710724482936711865e-01 7.43449702189565186927211470901966e-01 1.33950159398430468371543611283414e-02 7.64568774392989942284559745644401e-03
            2.43155281870591710724482936711865e-01 1.33950159398430850010708326180975e-02 7.43449702189565186927211470901966e-01 7.64568774392989248395169354921563e-03
            1.33950159398431006135821164093613e-02 7.43449702189565297949513933417620e-01 2.43155281870591655213331705454038e-01 7.64568774392990375965428739846175e-03
            1.33950159398430676538360728500265e-02 2.43155281870591710724482936711865e-01 7.43449702189565186927211470901966e-01 7.64568774392989508603690751442628e-03
            6.22112357616232514168075340421638e-01 3.65088603079598228440261209470918e-01 1.27990393041692573916634501074441e-02 8.41111285245294167112195538038577e-03
            6.22112357616232292123470415390329e-01 1.27990393041693493320076768782201e-02 3.65088603079598339462563671986572e-01 8.41111285245293820167500342677158e-03
            3.65088603079598394973714903244399e-01 6.22112357616232292123470415390329e-01 1.27990393041693684139659126230981e-02 8.41111285245294861001585928761415e-03
            3.65088603079598450484866134502226e-01 1.27990393041693961695415282520116e-02 6.22112357616232181101167952874675e-01 8.41111285245296248780366710207090e-03
            1.27990393041693597403485327390626e-02 6.22112357616232403145772877905983e-01 3.65088603079598228440261209470918e-01 8.41111285245295901835671514845671e-03
            1.27990393041693614750720087158697e-02 3.65088603079598394973714903244399e-01 6.22112357616232181101167952874675e-01 8.41111285245294861001585928761415e-03
            5.17329851757723613303596721380018e-01 4.12641979622127230697969935135916e-01 7.00281686201491559984333434840664e-02 1.68342619500367815521535419520660e-02
            5.17329851757723613303596721380018e-01 7.00281686201491559984333434840664e-02 4.12641979622127230697969935135916e-01 1.68342619500367780827065899984518e-02
            4.12641979622127286209121166393743e-01 5.17329851757723613303596721380018e-01 7.00281686201491559984333434840664e-02 1.68342619500367884910474458592944e-02
            4.12641979622127286209121166393743e-01 7.00281686201491698762211512985232e-02 5.17329851757723613303596721380018e-01 1.68342619500367850216004939056802e-02
            7.00281686201491282428577278551529e-02 5.17329851757723613303596721380018e-01 4.12641979622127230697969935135916e-01 1.68342619500367676743657341376093e-02
            7.00281686201491421206455356696097e-02 4.12641979622127286209121166393743e-01 5.17329851757723613303596721380018e-01 1.68342619500367780827065899984518e-02
            7.62594448273204550403647772327531e-01 1.66359403543997558250211454833334e-01 7.10461481827978913461407728391350e-02 1.77605299356818416289804929419915e-02
            7.62594448273204661425950234843185e-01 7.10461481827977525682626946945675e-02 1.66359403543997586005787070462247e-01 1.77605299356818277511926851275348e-02
            1.66359403543997613761362686091161e-01 7.62594448273204550403647772327531e-01 7.10461481827978635905651572102215e-02 1.77605299356818381595335409883774e-02
            1.66359403543997502739060223575507e-01 7.10461481827978219572017337668512e-02 7.62594448273204661425950234843185e-01 1.77605299356818381595335409883774e-02
            7.10461481827977109348992712511972e-02 7.62594448273204772448252697358839e-01 1.66359403543997474983484607946593e-01 1.77605299356818069345109734058497e-02
            7.10461481827977803238383103234810e-02 1.66359403543997502739060223575507e-01 7.62594448273204772448252697358839e-01 1.77605299356818277511926851275348e-02
            6.40898402664594746269699498952832e-01 2.92115957257160496762793400193914e-01 6.69856400782447569675071008532541e-02 1.93915333518244817345710373501788e-02
            6.40898402664594857292001961468486e-01 6.69856400782446320674168305231433e-02 2.92115957257160496762793400193914e-01 1.93915333518244539789954217212653e-02
            2.92115957257160496762793400193914e-01 6.40898402664594746269699498952832e-01 6.69856400782447014563558695954271e-02 1.93915333518244678567832295357221e-02
            2.92115957257160441251642168936087e-01 6.69856400782447292119314852243406e-02 6.40898402664594857292001961468486e-01 1.93915333518244886734649412574072e-02
            6.69856400782447847230827164821676e-02 6.40898402664594746269699498952832e-01 2.92115957257160441251642168936087e-01 1.93915333518245025512527490718639e-02
            6.69856400782447014563558695954271e-02 2.92115957257160441251642168936087e-01 6.40898402664594857292001961468486e-01 1.93915333518244678567832295357221e-02
            5.57643332782426282001608797145309e-01 2.81112663730934020378526838612743e-01 1.61244003486639697619864364241948e-01 2.88809571685944600771822621254614e-02
            5.57643332782426170979306334629655e-01 1.61244003486639475575259439210640e-01 2.81112663730934353445434226159705e-01 2.88809571685944600771822621254614e-02
            2.81112663730934131400829301128397e-01 5.57643332782426282001608797145309e-01 1.61244003486639586597561901726294e-01 2.88809571685944635466292140790756e-02
            2.81112663730934075889678069870570e-01 1.61244003486639669864288748613035e-01 5.57643332782426282001608797145309e-01 2.88809571685944392605005504037763e-02
            1.61244003486639697619864364241948e-01 5.57643332782426170979306334629655e-01 2.81112663730934131400829301128397e-01 2.88809571685944288521596945429337e-02
            1.61244003486639558841986286097381e-01 2.81112663730934242423131763644051e-01 5.57643332782426170979306334629655e-01 2.88809571685944357910535984501621e-02
            ];
        
    case  19
        
        % ALG. DEG.:   19
        % PTS CARD.:   78
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 8.5e-16
        
        xyw_bar=[
            1.19057700723167961959214622424952e-02 1.19057700723167684403458466135817e-02 9.76188459855366508222118682169821e-01 1.78549331856358954595531685072274e-03
            1.19057700723167302764293751238256e-02 9.76188459855366508222118682169821e-01 1.19057700723167458889406589150894e-02 1.78549331856357675236968152177042e-03
            9.76188459855366397199816219654167e-01 1.19057700723167441542171829382823e-02 1.19057700723168586459665974075506e-02 1.78549331856358585966793040000766e-03
            4.66749672319444697254908760442049e-01 4.66749672319444475210303835410741e-01 6.65006553611108830459386354050366e-02 7.21609522839577761238460951176421e-03
            4.66749672319444475210303835410741e-01 6.65006553611108830459386354050366e-02 4.66749672319444697254908760442049e-01 7.21609522839587735898447817817214e-03
            6.65006553611110773349679448074312e-02 4.66749672319444475210303835410741e-01 4.66749672319444475210303835410741e-01 7.21609522839581143949239105950255e-03
            5.44172191454611181571365818854247e-02 5.44172191454610001959402154625423e-02 8.91165561709077791441302451858064e-01 8.34902453149461050430080177875425e-03
            5.44172191454610695848792545348260e-02 8.91165561709077902463604914373718e-01 5.44172191454610487681975428131409e-02 8.34902453149459489178951798749040e-03
            8.91165561709077791441302451858064e-01 5.44172191454611250960304857926531e-02 5.44172191454610834626670623492828e-02 8.34902453149459489178951798749040e-03
            1.49494614598017633300131024043367e-01 1.49494614598017772078009102187934e-01 7.01010770803964566866284258139785e-01 1.88629245400943924837466880717329e-02
            1.49494614598017494522252945898799e-01 7.01010770803965010955494108202402e-01 1.49494614598017494522252945898799e-01 1.88629245400943716670649763500478e-02
            7.01010770803965010955494108202402e-01 1.49494614598017494522252945898799e-01 1.49494614598017494522252945898799e-01 1.88629245400943647281710724428194e-02
            2.45242132245316490912756535180961e-01 2.45242132245316490912756535180961e-01 5.09515735509367129196789392153732e-01 2.97327779925530069626837814666942e-02
            2.45242132245316657446210228954442e-01 5.09515735509366574085277079575462e-01 2.45242132245316768468512691470096e-01 2.97327779925529722682142619305523e-02
            5.09515735509366574085277079575462e-01 2.45242132245316685201785844583355e-01 2.45242132245316740712937075841182e-01 2.97327779925529618598734060697097e-02
            3.74619062916066836166351095016580e-01 3.74619062916066891677502326274407e-01 2.50761874167866272156146578709013e-01 3.18997491833587290543938763676124e-02
            3.74619062916066725144048632500926e-01 2.50761874167866383178449041224667e-01 3.74619062916066891677502326274407e-01 3.18997491833587290543938763676124e-02
            2.50761874167866327667297809966840e-01 3.74619062916066725144048632500926e-01 3.74619062916066947188653557532234e-01 3.18997491833587429321816841820691e-02
            9.33764288367549766256558996246895e-01 5.69009106573930140138273259253765e-02 9.33480097505721972961367782772868e-03 2.81688502938362344760192534920407e-03
            9.33764288367549544211954071215587e-01 9.33480097505729258799966885362664e-03 5.69009106573931666694932118844008e-02 2.81688502938363775907060215786260e-03
            5.69009106573929862582517102964630e-02 9.33764288367549766256558996246895e-01 9.33480097505723360740148564218543e-03 2.81688502938362778441061529122180e-03
            5.69009106573932499362200587711413e-02 9.33480097505718503514415829158679e-03 9.33764288367549544211954071215587e-01 2.81688502938364339692189908248565e-03
            9.33480097505724401574234150302800e-03 9.33764288367549766256558996246895e-01 5.69009106573930001360395181109197e-02 2.81688502938363082017669825063422e-03
            9.33480097505720932127282196688611e-03 5.69009106573930556471907493687468e-02 9.33764288367549655234256533731241e-01 2.81688502938362821809148428542358e-03
            8.65595106103016709298003661388066e-01 1.21539313278482469238284124912752e-01 1.28655806185008214637122136991820e-02 4.77873693712922448934010688503804e-03
            8.65595106103016709298003661388066e-01 1.28655806185009307512912002380290e-02 1.21539313278482358215981662397098e-01 4.77873693712920280529665717494936e-03
            1.21539313278482413727132893654925e-01 8.65595106103016709298003661388066e-01 1.28655806185008492192878293280955e-02 4.77873693712922535670184487344159e-03
            1.21539313278482996594220821862109e-01 1.28655806185009706499311477045921e-02 8.65595106103016043164188886294141e-01 4.77873693712925571436267446756574e-03
            1.28655806185008856484808248410445e-02 8.65595106103016709298003661388066e-01 1.21539313278482441482708509283839e-01 4.77873693712922101989315493142385e-03
            1.28655806185009168735033924235722e-02 1.21539313278482399849345085840469e-01 8.65595106103016598275701198872412e-01 4.77873693712922882614879682705578e-03
            7.80245213926672342985568775475258e-01 2.08294353280180810639166111286613e-01 1.14604327931468463752651132381288e-02 6.00142486301635889184291627884704e-03
            7.80245213926672898097081088053528e-01 1.14604327931468758655642048438494e-02 2.08294353280180227772078183079429e-01 6.00142486301634241196989449917965e-03
            2.08294353280180616350136801884219e-01 7.80245213926672454007871237990912e-01 1.14604327931469018864163444959559e-02 6.00142486301636496337508219767187e-03
            2.08294353280181088194922267575748e-01 1.14604327931468602530529210525856e-02 7.80245213926672009918661387928296e-01 6.00142486301632506473513473110870e-03
            1.14604327931468619877763970293927e-02 7.80245213926672565030173700506566e-01 2.08294353280180533083409954997478e-01 6.00142486301635715711944030203995e-03
            1.14604327931468151502425456556011e-02 2.08294353280180505327834339368565e-01 7.80245213926672676052476163022220e-01 6.00142486301634935086379840640802e-03
            5.47138472365763073312905362399761e-01 4.41297885394377531476095555262873e-01 1.15636422398593952109990823373664e-02 6.47131688392673602022719947512996e-03
            5.47138472365763073312905362399761e-01 1.15636422398593518429121829171891e-02 4.41297885394377586987246786520700e-01 6.47131688392678285776105084892151e-03
            4.41297885394377475964944324005046e-01 5.47138472365763184335207824915415e-01 1.15636422398593952109990823373664e-02 6.47131688392673515286546148672642e-03
            4.41297885394377087386885705200257e-01 1.15636422398593483734652309635749e-02 5.47138472365763517402115212462377e-01 6.47131688392673168341850953311223e-03
            1.15636422398593657206999907316458e-02 5.47138472365763073312905362399761e-01 4.41297885394377531476095555262873e-01 6.47131688392674556120631734756898e-03
            1.15636422398593310262304711955039e-02 4.41297885394377475964944324005046e-01 5.47138472365763295357510287431069e-01 6.47131688392673255078024752151578e-03
            6.65889139137802787971054385707248e-01 3.23322608813239320557642031417345e-01 1.07882520489578914713035828754073e-02 6.76709939308820125625221209020310e-03
            6.65889139137803343082566698285518e-01 1.07882520489579938199886655070259e-02 3.23322608813238654423827256323420e-01 6.76709939308821860348697185827405e-03
            3.23322608813239098513037106386037e-01 6.65889139137802898993356848222902e-01 1.07882520489580580047572766488884e-02 6.76709939308821079723132996264212e-03
            3.23322608813239320557642031417345e-01 1.07882520489579730033069537853407e-02 6.65889139137802787971054385707248e-01 6.76709939308817176595312048448250e-03
            1.07882520489579695338600018317265e-02 6.65889139137802898993356848222902e-01 3.23322608813239154024188337643864e-01 6.76709939308820646042264002062439e-03
            1.07882520489579521866252420636556e-02 3.23322608813239209535339568901691e-01 6.65889139137802787971054385707248e-01 6.76709939308819431735830818297472e-03
            7.97030463292363489102854146040045e-01 1.39071275726620063872829291540256e-01 6.38982609810164470243165624196990e-02 1.42554774818164364802441923529841e-02
            7.97030463292363711147459071071353e-01 6.38982609810165302910434093064396e-02 1.39071275726619758561497519622208e-01 1.42554774818164226024563845385273e-02
            1.39071275726619980606102444653516e-01 7.97030463292363489102854146040045e-01 6.38982609810165858021946405642666e-02 1.42554774818164364802441923529841e-02
            1.39071275726620174895131754055910e-01 6.38982609810166551911336796365504e-02 7.97030463292363156035946758493083e-01 1.42554774818164555622024280978621e-02
            6.38982609810165441688312171208963e-02 7.97030463292363600125156608555699e-01 1.39071275726619814072648750880035e-01 1.42554774818164295413502884457557e-02
            6.38982609810166274355580640076369e-02 1.39071275726619952850526829024602e-01 7.97030463292363489102854146040045e-01 1.42554774818164468885850482138267e-02
            6.88298973534611890912060516711790e-01 2.54674408859317524811416433294653e-01 5.70266176060705842765230499935569e-02 1.60037930469036603076116165311760e-02
            6.88298973534612446023572829290060e-01 5.70266176060704593764327796634461e-02 2.54674408859317080722206583232037e-01 1.60037930469036221436951450414199e-02
            2.54674408859317413789113970778999e-01 6.88298973534612001934362979227444e-01 5.70266176060706397876742812513839e-02 1.60037930469036533687177126239476e-02
            2.54674408859317469300265202036826e-01 5.70266176060703552930242210550205e-02 6.88298973534612223978967904258752e-01 1.60037930469036152048012411341915e-02
            5.70266176060704593764327796634461e-02 6.88298973534612223978967904258752e-01 2.54674408859317358277962739521172e-01 1.60037930469036394909299048094908e-02
            5.70266176060703622319181249622488e-02 2.54674408859317413789113970778999e-01 6.88298973534612335001270366774406e-01 1.60037930469036152048012411341915e-02
            5.57143388414403584185663476091577e-01 3.85518010456119386564211026779958e-01 5.73386011294770292501254971284652e-02 1.63620916871316801621816239276086e-02
            5.57143388414403917252570863638539e-01 5.73386011294770847612767283862922e-02 3.85518010456118997986152407975169e-01 1.63620916871316871010755278348370e-02
            3.85518010456119386564211026779958e-01 5.57143388414403584185663476091577e-01 5.73386011294770847612767283862922e-02 1.63620916871317079177572395565221e-02
            3.85518010456119386564211026779958e-01 5.73386011294770708834889205718355e-02 5.57143388414403584185663476091577e-01 1.63620916871316558760529602523093e-02
            5.73386011294770500668072088501503e-02 5.57143388414403695207965938607231e-01 3.85518010456119220030757333006477e-01 1.63620916871317148566511434637505e-02
            5.73386011294770292501254971284652e-02 3.85518010456119386564211026779958e-01 5.57143388414403695207965938607231e-01 1.63620916871317009788633356492937e-02
            6.12834567107272132346906801103614e-01 2.52827382426030478068668116975459e-01 1.34338050466697389584425081920926e-01 2.04886348693677500243381928157760e-02
            6.12834567107272132346906801103614e-01 1.34338050466697306317698235034186e-01 2.52827382426030533579819348233286e-01 2.04886348693677673715729525838469e-02
            2.52827382426030478068668116975459e-01 6.12834567107272132346906801103614e-01 1.34338050466697334073273850663099e-01 2.04886348693677534937851447693902e-02
            2.52827382426030589090970579491113e-01 1.34338050466696973250790847487224e-01 6.12834567107272465413814188650576e-01 2.04886348693677257382095291404767e-02
            1.34338050466697250806547003776359e-01 6.12834567107272132346906801103614e-01 2.52827382426030644602121810748940e-01 2.04886348693677534937851447693902e-02
            1.34338050466696973250790847487224e-01 2.52827382426030811135575504522421e-01 6.12834567107272132346906801103614e-01 2.04886348693677153298686732796341e-02
            4.92042973154920293410441445303150e-01 3.66154971066125212164621416377486e-01 1.41802055778954494424937138319365e-01 2.37981740776729878972250986635117e-02
            4.92042973154920182388138982787495e-01 1.41802055778954466669361522690451e-01 3.66154971066125378698075110150967e-01 2.37981740776729878972250986635117e-02
            3.66154971066125267675772647635313e-01 4.92042973154920348921592676560977e-01 1.41802055778954383402634675803711e-01 2.37981740776729844277781467098976e-02
            3.66154971066125101142318953861832e-01 1.41802055778954577691663985206105e-01 4.92042973154920293410441445303150e-01 2.37981740776729670805433869418266e-02
            1.41802055778954522180512753948278e-01 4.92042973154920237899290214045323e-01 3.66154971066125212164621416377486e-01 2.37981740776729532027555791273699e-02
            1.41802055778954522180512753948278e-01 3.66154971066125212164621416377486e-01 4.92042973154920237899290214045323e-01 2.37981740776729462638616752201415e-02
            ];
        
    case  20
        
        % ALG. DEG.:   20
        % PTS CARD.:   91
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.3e-16
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333425851918718763045e-01 2.79414564705804009669964926843022e-02
            8.57077238529909204989287729858916e-03 8.57077238529917878606667613894388e-03 9.82858455229401739572381302423310e-01 9.90558155838326291273143731075379e-04
            8.57077238529914756104410855641618e-03 9.82858455229401739572381302423310e-01 8.57077238529913021380934878834523e-03 9.90558155838330844922268170194002e-04
            9.82858455229401850594683764938964e-01 8.57077238529911286657458902027429e-03 8.57077238529903653874164604076213e-03 9.90558155838328242837054204983360e-04
            5.15305405051263848426046365602815e-02 5.15305405051261489202119037145167e-02 8.96938918989747424603820036281832e-01 6.40849365050742865446142815244457e-03
            5.15305405051262738203021740446275e-02 8.96938918989747424603820036281832e-01 5.15305405051262876980899818590842e-02 6.40849365050742345029100022202329e-03
            8.96938918989747424603820036281832e-01 5.15305405051263085147716935807694e-02 5.15305405051262668814082701373991e-02 6.40849365050743732807880803648004e-03
            4.94659340413900083799347839885741e-01 4.94659340413900083799347839885741e-01 1.06813191721997768901530889706919e-02 6.96401200751685812290192600926275e-03
            4.94659340413900083799347839885741e-01 1.06813191721997491345774733417784e-02 4.94659340413900139310499071143568e-01 6.96401200751684337775238020640245e-03
            1.06813191721997803596000409243061e-02 4.94659340413900083799347839885741e-01 4.94659340413900083799347839885741e-01 6.96401200751685899026366399766630e-03
            4.37325812915608291842062271825853e-01 4.37325812915608291842062271825853e-01 1.25348374168783360804724225090467e-01 1.66734061189873564168273389896058e-02
            4.37325812915608291842062271825853e-01 1.25348374168783277537997378203727e-01 4.37325812915608347353213503083680e-01 1.66734061189873217223578194534639e-02
            1.25348374168783444071451071977208e-01 4.37325812915608069797457346794545e-01 4.37325812915608513886667196857161e-01 1.66734061189873494779334350823774e-02
            1.32293995381116469278737213244312e-01 1.32293995381116663567766522646707e-01 7.35412009237766950420223110995721e-01 1.67514387916361633112938278600268e-02
            1.32293995381116608056615291388880e-01 7.35412009237766839397920648480067e-01 1.32293995381116524789888444502139e-01 1.67514387916361598418468759064126e-02
            7.35412009237766839397920648480067e-01 1.32293995381116497034312828873226e-01 1.32293995381116663567766522646707e-01 1.67514387916361702501877317672552e-02
            2.15238824344608503658093923149863e-01 2.15238824344608420391367076263123e-01 5.69522351310783103706114616215928e-01 2.24256979453981500349790678683348e-02
            2.15238824344608559169245154407690e-01 5.69522351310782992683812153700273e-01 2.15238824344608392635791460634209e-01 2.24256979453981812600016354508625e-02
            5.69522351310782992683812153700273e-01 2.15238824344608392635791460634209e-01 2.15238824344608614680396385665517e-01 2.24256979453981604433199237291774e-02
            9.43408146097875710545110905513866e-01 4.66217699188314349534323355328524e-02 9.97008398329285450145675895328168e-03 2.73385455951450501155264305452874e-03
            9.43408146097875377478203517966904e-01 9.97008398329295511541836560809315e-03 4.66217699188316708758250683786173e-02 2.73385455951451455253176092696776e-03
            4.66217699188315945479921253991051e-02 9.43408146097875488500505980482558e-01 9.97008398329291001260799021110870e-03 2.73385455951450978204220199074825e-03
            4.66217699188313447478115847388835e-02 9.97008398329291521677841814152998e-03 9.43408146097875710545110905513866e-01 2.73385455951449937370134612990569e-03
            9.97008398329294991124793767767187e-03 9.43408146097875488500505980482558e-01 4.66217699188316014868860293063335e-02 2.73385455951450891468046400234471e-03
            9.97008398329293776818360584002221e-03 4.66217699188316778147189722858457e-02 9.43408146097875377478203517966904e-01 2.73385455951451628725523690377486e-03
            7.95199869823192462270355918008136e-01 1.94613925044525570839937245182227e-01 1.01862051322819668897068368096370e-02 3.87123692869166492400156975861591e-03
            7.95199869823192462270355918008136e-01 1.01862051322818593368513262475972e-02 1.94613925044525681862239707697881e-01 3.87123692869164280627725105432546e-03
            1.94613925044525598595512860811141e-01 7.95199869823192462270355918008136e-01 1.01862051322819668897068368096370e-02 3.87123692869166318927809378180882e-03
            1.94613925044525598595512860811141e-01 1.01862051322819217868964614126526e-02 7.95199869823192462270355918008136e-01 3.87123692869167229657634266004607e-03
            1.01862051322819582160894569256016e-02 7.95199869823192462270355918008136e-01 1.94613925044525570839937245182227e-01 3.87123692869166926081025970063365e-03
            1.01862051322819096438321295750029e-02 1.94613925044525598595512860811141e-01 7.95199869823192573292658380523790e-01 3.87123692869165885246940383979108e-03
            7.24428132024519610965285210113507e-01 2.65496289314309019680138135299785e-01 1.00755786611713693545766545867082e-02 4.15623380054164041907949211918094e-03
            7.24428132024519610965285210113507e-01 1.00755786611713450684479909114089e-02 2.65496289314309019680138135299785e-01 4.15623380054164302116470608439158e-03
            2.65496289314309075191289366557612e-01 7.24428132024519610965285210113507e-01 1.00755786611712583322741920710541e-02 4.15623380054164302116470608439158e-03
            2.65496289314309075191289366557612e-01 1.00755786611713745587470825171295e-02 7.24428132024519499942982747597853e-01 4.15623380054163001073863625833837e-03
            1.00755786611713173128723752824953e-02 7.24428132024519610965285210113507e-01 2.65496289314309019680138135299785e-01 4.15623380054164388852644407279513e-03
            1.00755786611713762934705584939366e-02 2.65496289314309075191289366557612e-01 7.24428132024519610965285210113507e-01 4.15623380054165169478208596842705e-03
            8.75117681769159161397908519575140e-01 1.14200622336593365924173326675373e-01 1.06816958942474726779181537494878e-02 4.31374920999024955015066495889187e-03
            8.75117681769159050375606057059485e-01 1.06816958942474518612364420278027e-02 1.14200622336593504702051404819940e-01 4.31374920999025215223587892410251e-03
            1.14200622336593421435324557933200e-01 8.75117681769159161397908519575140e-01 1.06816958942473894111913068627473e-02 4.31374920999023827444807110964575e-03
            1.14200622336593227146295248530805e-01 1.06816958942474882904294375407517e-02 8.75117681769159272420210982090794e-01 4.31374920999025735640630685452379e-03
            1.06816958942474431876190621437672e-02 8.75117681769159161397908519575140e-01 1.14200622336593449190900173562113e-01 4.31374920999024174389502306325994e-03
            1.06816958942474189014903984684679e-02 1.14200622336593490824263597005483e-01 8.75117681769159161397908519575140e-01 4.31374920999023133555416720241737e-03
            6.21242229278482605003830485657090e-01 3.69462178934587270884293275230448e-01 9.29559178693012411187623911246192e-03 5.77437855793759471229043001017089e-03
            6.21242229278482493981528023141436e-01 9.29559178693014492855795083414705e-03 3.69462178934587381906595737746102e-01 5.77437855793759297756695403336380e-03
            3.69462178934587437417746969003929e-01 6.21242229278482382959225560625782e-01 9.29559178693017962302747037028894e-03 5.77437855793758864075826409134606e-03
            3.69462178934587326395444506488275e-01 9.29559178693014492855795083414705e-03 6.21242229278482493981528023141436e-01 5.77437855793758777339652610294252e-03
            9.29559178693013798966404692691867e-03 6.21242229278482382959225560625782e-01 3.69462178934587437417746969003929e-01 5.77437855793758864075826409134606e-03
            9.29559178693011890770581118204063e-03 3.69462178934587381906595737746102e-01 6.21242229278482493981528023141436e-01 5.77437855793758517131131213773187e-03
            8.23284162466894064991151935828384e-01 1.21432174389561192695552449549723e-01 5.52836631435447423132956146218930e-02 9.99979838740234702743059358454047e-03
            8.23284162466894287035756860859692e-01 5.52836631435446521076748638279241e-02 1.21432174389561053917674371405155e-01 9.99979838740235396632449749176885e-03
            1.21432174389561137184401218291896e-01 8.23284162466894287035756860859692e-01 5.52836631435446035354175364773255e-02 9.99979838740236263994187737580432e-03
            1.21432174389561234328915872993093e-01 5.52836631435447006799321911785228e-02 8.23284162466894064991151935828384e-01 9.99979838740236437466535335261142e-03
            5.52836631435446382298870560134674e-02 8.23284162466894287035756860859692e-01 1.21432174389561109428825602662982e-01 9.99979838740236090521840139899723e-03
            5.52836631435445410853724013122701e-02 1.21432174389561012284310947961785e-01 8.23284162466894509080361785891000e-01 9.99979838740235570104797346857595e-03
            7.35502973872642962227530460950220e-01 2.11973966375945732654173525588703e-01 5.25230597514113051182960134610767e-02 1.15386309756824092381810586971369e-02
            7.35502973872642962227530460950220e-01 5.25230597514112565460386861104780e-02 2.11973966375945788165324756846530e-01 1.15386309756824317895862463956291e-02
            2.11973966375945843676475988104357e-01 7.35502973872642851205227998434566e-01 5.25230597514112496071447822032496e-02 1.15386309756824231159688665115937e-02
            2.11973966375945788165324756846530e-01 5.25230597514113189960838212755334e-02 7.35502973872642851205227998434566e-01 1.15386309756824075034575827203298e-02
            5.25230597514113467516594369044469e-02 7.35502973872642851205227998434566e-01 2.11973966375945788165324756846530e-01 1.15386309756824317895862463956291e-02
            5.25230597514113745072350525333604e-02 2.11973966375945788165324756846530e-01 7.35502973872642851205227998434566e-01 1.15386309756824456673740542100859e-02
            5.24604158534802356150805735524045e-01 4.22476993945888334991423107567243e-01 5.29188475193093088577711569087114e-02 1.18484310488777347392108652002207e-02
            5.24604158534802356150805735524045e-01 5.29188475193092394688321178364276e-02 4.22476993945888390502574338825070e-01 1.18484310488777156572526294553427e-02
            4.22476993945888390502574338825070e-01 5.24604158534802356150805735524045e-01 5.29188475193093088577711569087114e-02 1.18484310488777225961465333625711e-02
            4.22476993945888390502574338825070e-01 5.29188475193092117132565022075141e-02 5.24604158534802467173108198039699e-01 1.18484310488777278003169612929923e-02
            5.29188475193092811021955412797979e-02 5.24604158534802134106200810492737e-01 4.22476993945888557036028032598551e-01 1.18484310488777260655934853161853e-02
            5.29188475193092811021955412797979e-02 4.22476993945888390502574338825070e-01 5.24604158534802245128503273008391e-01 1.18484310488776879016770138264292e-02
            6.30873630515176087030226881324779e-01 3.19958418028188196213790206456906e-01 4.91679514566357167559829122183146e-02 1.21077889838240183167927810359288e-02
            6.30873630515175976007924418809125e-01 4.91679514566356820615133926821727e-02 3.19958418028188362747243900230387e-01 1.21077889838240217862397329895430e-02
            3.19958418028188418258395131488214e-01 6.30873630515175976007924418809125e-01 4.91679514566355502225292184448335e-02 1.21077889838240096431754011518933e-02
            3.19958418028188251724941437714733e-01 4.91679514566356473670438731460308e-02 6.30873630515176087030226881324779e-01 1.21077889838240165820693050591217e-02
            4.91679514566356334892560653315741e-02 6.30873630515175976007924418809125e-01 3.19958418028188362747243900230387e-01 1.21077889838240321945805888503855e-02
            4.91679514566356612448316809604876e-02 3.19958418028188529280697594003868e-01 6.30873630515175864985621956293471e-01 1.21077889838240183167927810359288e-02
            6.45556379414073733258305765048135e-01 2.34468421561538770125920905229577e-01 1.19975199024387496615773329722288e-01 1.73003035754914734345444315977147e-02
            6.45556379414073733258305765048135e-01 1.19975199024387316204531828134350e-01 2.34468421561538964414950214631972e-01 1.73003035754914769039913835513289e-02
            2.34468421561538908903798983374145e-01 6.45556379414073733258305765048135e-01 1.19975199024387357837895251577720e-01 1.73003035754914526178627198760296e-02
            2.34468421561538742370345289600664e-01 1.19975199024387357837895251577720e-01 6.45556379414073955302910690079443e-01 1.73003035754914595567566237832580e-02
            1.19975199024387454982409906278917e-01 6.45556379414073733258305765048135e-01 2.34468421561538797881496520858491e-01 1.73003035754914699650974796441005e-02
            1.19975199024387468860197714093374e-01 2.34468421561538964414950214631972e-01 6.45556379414073511213700840016827e-01 1.73003035754914560873096718296438e-02
            5.39652455990818680220399983227253e-01 3.37729237107069868439879201105214e-01 1.22618306902111451339720815667533e-01 1.76675135330825149038158627945450e-02
            5.39652455990818458175795058195945e-01 1.22618306902111423584145200038620e-01 3.37729237107070145995635357394349e-01 1.76675135330824802093463432584031e-02
            3.37729237107069923951030432363041e-01 5.39652455990818680220399983227253e-01 1.22618306902111395828569584409706e-01 1.76675135330825044954750069337024e-02
            3.37729237107069923951030432363041e-01 1.22618306902111354195206160966336e-01 5.39652455990818680220399983227253e-01 1.76675135330825149038158627945450e-02
            1.22618306902111340317418353151879e-01 5.39652455990818680220399983227253e-01 3.37729237107069923951030432363041e-01 1.76675135330825322510506225626159e-02
            1.22618306902111395828569584409706e-01 3.37729237107069923951030432363041e-01 5.39652455990818680220399983227253e-01 1.76675135330824593926646315367179e-02
            4.51386500463365036850404976576101e-01 3.28819489995002267601620360437664e-01 2.19794009541632695547974662986235e-01 2.55910343589249751616865324876926e-02
            4.51386500463364925828102514060447e-01 2.19794009541632556770096584841667e-01 3.28819489995002545157376516726799e-01 2.55910343589249682227926285804642e-02
            3.28819489995002489646225285468972e-01 4.51386500463364925828102514060447e-01 2.19794009541632529014520969212754e-01 2.55910343589249786311334844413068e-02
            3.28819489995002434135074054211145e-01 2.19794009541632473503369737954927e-01 4.51386500463365092361556207833928e-01 2.55910343589249612838987246732358e-02
            2.19794009541632556770096584841667e-01 4.51386500463365036850404976576101e-01 3.28819489995002434135074054211145e-01 2.55910343589250063867091000702203e-02
            2.19794009541632640036823431728408e-01 3.28819489995002323112771591695491e-01 4.51386500463365036850404976576101e-01 2.55910343589249959783682442093777e-02
            ];
        
    case  22
        
        % ALG. DEG.:   22
        % PTS CARD.:  105
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.7e-16
        
        xyw_bar=[
            5.81913467082437706190400916739236e-03 5.81913467082493390813979772246967e-03 9.88361730658350645661869293689961e-01 4.68637034057991499688688996627661e-04
            5.81913467082731741819578985541739e-03 9.88361730658345760680560943001183e-01 5.81913467082695312626583472592756e-03 4.68637034058211863780246675403873e-04
            9.88361730658347092948190493189031e-01 5.81913467082685945119813197834446e-03 5.81913467082604760061137483262428e-03 4.68637034058137324880888296974035e-04
            3.70242529014878085646955696574878e-02 3.70242529015019708471534443106066e-02 9.25951494197010172015893658681307e-01 3.33907489813184150256875071249851e-03
            3.70242529014817092769540352037438e-02 9.25951494197024382870608860685024e-01 3.70242529014939147913310080184601e-02 3.33907489813209217011102936112366e-03
            9.25951494197024382870608860685024e-01 3.70242529014770810347201290824160e-02 3.70242529014985360946710102325596e-02 3.33907489813192216721038363402840e-03
            1.00139964243313464575813043211383e-01 1.00139964243318779768543436148320e-01 7.99720071513367658511128865939099e-01 1.00945212949187321932464911355964e-02
            1.00139964243327175830167163894657e-01 7.99720071513356778325487539405003e-01 1.00139964243316059722133104514796e-01 1.00945212949185014750241862202529e-02
            7.99720071513358554682326939655468e-01 1.00139964243326967663350046677806e-01 1.00139964243314477654323013666726e-01 1.00945212949183887179982477277917e-02
            4.77283361444081621893786859800457e-01 4.77283361444104714532699063056498e-01 4.54332771118136080623628458852181e-02 1.15798841292788484008680072179232e-02
            4.77283361444093778835906505264575e-01 4.54332771118138301069677709165262e-02 4.77283361444092391057125723818899e-01 1.15798841292778838946153641131787e-02
            4.54332771118135872456811341635330e-02 4.77283361444090392655681398537126e-01 4.77283361444096054793106986835483e-01 1.15798841292768898980636294027136e-02
            1.87840033887680113444673679623520e-01 1.87840033887668983458851812429202e-01 6.24319932224650875340898892318364e-01 1.72705278027751978064951998703691e-02
            1.87840033887670343482056978245964e-01 6.24319932224627338612776838999707e-01 1.87840033887702317905166182754328e-01 1.72705278027764572157387590323196e-02
            6.24319932224634666084739365032874e-01 1.87840033887671259416052294000110e-01 1.87840033887694074499208340967016e-01 1.72705278027763531323302004238940e-02
            4.02759919721537684722534322645515e-01 4.02759919721540182724339729247731e-01 1.94480160556922132553125948106754e-01 2.21456711367156479364215471150601e-02
            4.02759919721531467473596421768889e-01 1.94480160556923437065179882665689e-01 4.02759919721545123216799311194336e-01 2.21456711367135211654399995495623e-02
            1.94480160556922188064277179364581e-01 4.02759919721522974267458039321355e-01 4.02759919721554837668264781314065e-01 2.21456711367137085155754050447285e-02
            2.95819836686845805129308928371756e-01 2.95819836686848303131114334973972e-01 4.08360326626305836228425505396444e-01 2.40506213328556713026973312707923e-02
            2.95819836686846526774274934723508e-01 4.08360326626306835429147668037331e-01 2.95819836686846637796577397239162e-01 2.40506213328561327391419411014795e-02
            4.08360326626305503161518117849482e-01 2.95819836686846415751972472207854e-01 2.95819836686848081086509409942664e-01 2.40506213328561153919071813334085e-02
            9.60353805601490417842569513595663e-01 3.24378361207835266832866238928545e-02 7.20835827772605547414386251148244e-03 1.37243742238504578233315722712859e-03
            9.60353805601490417842569513595663e-01 7.20835827771566187843754747177627e-03 3.24378361207939211463546769209643e-02 1.37243742238409016653832850352046e-03
            3.24378361207852336511869850710355e-02 9.60353805601490417842569513595663e-01 7.20835827772436932292521305498667e-03 1.37243742238480855889781739875843e-03
            3.24378361207898827101026029140485e-02 7.20835827772483769826372679290216e-03 9.60353805601485310816656237875577e-01 1.37243742238554494901336955337001e-03
            7.20835827771712251560431994334976e-03 9.60353805601490417842569513595663e-01 3.24378361207924292841653368668631e-02 1.37243742238409146758093548612578e-03
            7.20835827772210117198037337971073e-03 3.24378361207898827101026029140485e-02 9.60353805601487975351915338251274e-01 1.37243742238494278312677110420736e-03
            9.12769702185145415107569988322211e-01 7.96405637940718197365086439276638e-02 7.58973402078276515592136775012477e-03 2.19970879574913789553103349305729e-03
            9.12769702185116660331232196767814e-01 7.58973402078859816360933976397973e-03 7.96405637940947458419671534102235e-02 2.19970879575034439570857536239146e-03
            7.96405637940748173386751318503229e-02 9.12769702185143194661520738009131e-01 7.58973402078200187759193795500323e-03 2.19970879574896052005561486453189e-03
            7.96405637940890281933903338540404e-02 7.58973402078647225998953018688553e-03 9.12769702185124542914707035379251e-01 2.19970879574991028115871216641608e-03
            7.58973402078918363278248193637410e-03 9.12769702185126097226941510598408e-01 7.96405637940846844458064879290760e-02 2.19970879575023554181045781774628e-03
            7.58973402078367068157582764342806e-03 7.96405637940886951264829463070782e-02 9.12769702185127651539175985817565e-01 2.19970879574922983587526026383330e-03
            8.46930929612685123863968783552991e-01 1.44767660510349893687731537283980e-01 8.30140987696498244829967916302849e-03 3.13791477014358508382252388457800e-03
            8.46930929612685123863968783552991e-01 8.30140987696372303905612000107794e-03 1.44767660510351142688634240585088e-01 3.13791477014213615603921425645240e-03
            1.44767660510350892888453699924867e-01 8.46930929612682903417919533239910e-01 8.30140987696614818247553557739593e-03 3.13791477014385960381259721430069e-03
            1.44767660510350892888453699924867e-01 8.30140987696267700080010598640001e-03 8.46930929612686345109295871225186e-01 3.13791477014198393405419729162986e-03
            8.30140987696306210941177283757497e-03 8.46930929612694893826585484930547e-01 1.44767660510342066615407929930370e-01 3.13791477014225281619297369672950e-03
            8.30140987696398324757751652214210e-03 1.44767660510350892888453699924867e-01 8.46930929612685012841666321037337e-01 3.13791477014238161941106497465626e-03
            7.63800852902573046421252911386546e-01 2.27743920492919527109876298709423e-01 8.45522660450742646887078990403097e-03 3.98030762304541450846695482823634e-03
            7.63800852902583926606894237920642e-01 8.45522660450583572744331917192540e-03 2.27743920492910228992045063023397e-01 3.98030762304477613022779536322560e-03
            2.27743920492916557263285426415678e-01 7.63800852902575821978814474277897e-01 8.45522660450759300232448367751203e-03 3.98030762304541450846695482823634e-03
            2.27743920492911561259674613211246e-01 8.45522660450640992091386749507365e-03 7.63800852902582039227752375154523e-01 3.98030762304495914355451091637406e-03
            8.45522660450593807612840180354397e-03 7.63800852902583815584591775404988e-01 2.27743920492910256747620678652311e-01 3.98030762304509878879432704934516e-03
            8.45522660450685921429414548811110e-03 2.27743920492909812658410828589695e-01 7.63800852902583371495381925342372e-01 3.98030762304518032079769795927859e-03
            6.63853996353000908037245153536787e-01 3.27598731749433658144710079795914e-01 8.54727189756543381804476666729897e-03 4.66500380307044177491393455170510e-03
            6.63853996353002795416387016302906e-01 8.54727189756490125793764178752099e-03 3.27598731749432325877080529608065e-01 4.66500380307055539930161103256978e-03
            3.27598731749432103832475604576757e-01 6.63853996353002684394084553787252e-01 8.54727189756521177343984163599089e-03 4.66500380307046779576607420381151e-03
            3.27598731749430716053694823131082e-01 8.54727189756544596110909850494863e-03 6.63853996353003905639411641459446e-01 4.66500380307071672858487687562956e-03
            8.54727189756502442330443614082469e-03 6.63853996353001463148757466115057e-01 3.27598731749433547122407617280260e-01 4.66500380307046953048955018061861e-03
            8.54727189756525514152674105616825e-03 3.27598731749432103832475604576757e-01 6.63853996353002684394084553787252e-01 4.66500380307082514880212542607296e-03
            5.52855567624573107643470848415745e-01 4.38428235397557652941458172790590e-01 8.71619697786923941507097879366484e-03 5.07671711159572453669897029726599e-03
            5.52855567624569665952094510430470e-01 8.71619697786918216919627155903072e-03 4.38428235397561150143985742033692e-01 5.07671711159569851584683064515957e-03
            4.38428235397559484809448804298881e-01 5.52855567624571442308933910680935e-01 8.71619697786907288161728502018377e-03 5.07671711159575749644501385660078e-03
            4.38428235397557819474911866564071e-01 8.71619697786932962069172958763374e-03 5.52855567624572885598865923384437e-01 5.07671711159574708810415799575821e-03
            8.71619697786919604698407937348748e-03 5.52855567624573107643470848415745e-01 4.38428235397557708452609404048417e-01 5.07671711159552157405228101083594e-03
            8.71619697786918390391974753583781e-03 4.38428235397562870989673911026330e-01 5.52855567624567889595255110180005e-01 5.07671711159573494503982615810855e-03
            8.69853922593163075838162967556855e-01 8.94784746607361763937049659034528e-02 4.06676027461007477681320665396925e-02 5.88628834180328960185013542627530e-03
            8.69853922593129991192029137891950e-01 4.06676027461178452027112939504150e-02 8.94784746607521636052595681576349e-02 5.88628834180296867800707971696283e-03
            8.94784746607384245953298318454472e-02 8.69853922593161188459021104790736e-01 4.06676027461003730678612555493601e-02 5.88628834180289755434456466787196e-03
            8.94784746607328734802067060627451e-02 4.06676027461085748404556738933024e-02 8.69853922593158523923762004415039e-01 5.88628834180203539677700419474604e-03
            4.06676027461189415479481112924987e-02 8.69853922593127548701374962547561e-01 8.94784746607535375062525417888537e-02 5.88628834180388634672587144791578e-03
            4.06676027461031347476350106262544e-02 8.94784746607552861075163264104049e-02 8.69853922593141537511485239519970e-01 5.88628834180213601073861084955752e-03
            7.92000836049181700104782066773623e-01 1.64479570130585700526992809500371e-01 4.35195938202325993682251237260061e-02 8.24419693056494332084049858622166e-03
            7.92000836049191914156608618213795e-01 4.35195938202235649283622365146584e-02 1.64479570130584507037241337457090e-01 8.24419693056257195384883829092360e-03
            1.64479570130581009834713768213987e-01 7.92000836049181700104782066773623e-01 4.35195938202372900605041650123894e-02 8.24419693056463454006177471455885e-03
            1.64479570130567853691871960108983e-01 4.35195938202214555046154487172316e-02 7.92000836049210676925724783359328e-01 8.24419693056295879718398111890565e-03
            4.35195938202220175550216652027302e-02 7.92000836049181700104782066773623e-01 1.64479570130596330912453595374245e-01 8.24419693056336298775388371495865e-03
            4.35195938202279017370521785323945e-02 1.64479570130591612464598938458948e-01 7.92000836049180589881757441617083e-01 8.24419693056344451975725462489208e-03
            6.98032087133783618604354614944896e-01 2.57896711320231286634907519328408e-01 4.40712015459850947607378657266963e-02 9.92585710656785150107772608407686e-03
            6.98032087133790946076317140978063e-01 4.40712015459795644622964516656793e-02 2.57896711320229510278068119077943e-01 9.92585710656706393661963261365599e-03
            2.57896711320215854534865229652496e-01 6.98032087133799494793606754683424e-01 4.40712015459846506715280156640802e-02 9.92585710656689046427203493294655e-03
            2.57896711320220961560778505372582e-01 4.40712015459825898200385552172520e-02 6.98032087133796497191440266760765e-01 9.92585710656840487786656268553998e-03
            4.40712015459804318240344400692265e-02 6.98032087133772738418713288410800e-01 2.57896711320246829757252271519974e-01 9.92585710656723220479680236394415e-03
            4.40712015459831588093386756099790e-02 2.57896711320235505482401094923262e-01 6.98032087133781287136002902116161e-01 9.92585710656751322999991060669345e-03
            5.90921636898512403490713040810078e-01 3.64207294093871758455804865661776e-01 4.48710690076158380534820935281459e-02 1.11149932478954577286245353207050e-02
            5.90921636898516511315904153889278e-01 4.48710690076145682358976785053528e-02 3.64207294093868927387092071512598e-01 1.11149932478946389391438742677565e-02
            3.64207294093850664218336987687508e-01 5.90921636898534607951205543940887e-01 4.48710690076147278304574683716055e-02 1.11149932478948696573661791831000e-02
            3.64207294093867151030252671262133e-01 4.48710690076162890815858474979905e-02 5.90921636898516511315904153889278e-01 1.11149932478954820147531989960044e-02
            4.48710690076150608973648559185676e-02 5.90921636898503632728818502073409e-01 3.64207294093881306373816642008023e-01 1.11149932478939190289013438928123e-02
            4.48710690076153176364393004860176e-02 3.64207294093872813167678259560489e-01 5.90921636898511959401503190747462e-01 1.11149932478944203639859011900626e-02
            7.13756998577414591977685631718487e-01 1.81577875343951838926770392390608e-01 1.04665126078633569095543975890905e-01 1.31075965603976007239905854362405e-02
            7.13756998577440460174159397865878e-01 1.04665126078613612836676338702091e-01 1.81577875343945926989164263432031e-01 1.31075965603978036866372747226706e-02
            1.81577875343954003861668411445862e-01 7.13756998577404933037371392856585e-01 1.04665126078641090856535811326467e-01 1.31075965603982165508245572027590e-02
            1.81577875343944039610022400665912e-01 1.04665126078612918947285947979253e-01 7.13756998577443013687116035725921e-01 1.31075965603976579698652926708746e-02
            1.04665126078611864235412554080540e-01 7.13756998577441792441788948053727e-01 1.81577875343946315567222882236820e-01 1.31075965603975625600741139464844e-02
            1.04665126078623660355049196368782e-01 1.81577875343950506659140842202760e-01 7.13756998577425805230234345799545e-01 1.31075965603982061424837013419165e-02
            6.14456089155397067180786052631447e-01 2.79145603763534810859425761009334e-01 1.06398307081068121959788186359219e-01 1.55304540973436530637563635082188e-02
            6.14456089155397067180786052631447e-01 1.06398307081061002654642777542904e-01 2.79145603763541916286783362011192e-01 1.55304540973445603241342993783292e-02
            2.79145603763536809260870086291106e-01 6.14456089155397067180786052631447e-01 1.06398307081066123558343861077446e-01 1.55304540973434587747270541058242e-02
            2.79145603763538918684616874088533e-01 1.06398307081066303969585362665384e-01 6.14456089155394735712434339802712e-01 1.55304540973445447116230155870653e-02
            1.06398307081062487577938213689777e-01 6.14456089155397067180786052631447e-01 2.79145603763540472996851349307690e-01 1.55304540973443053197833307876863e-02
            1.06398307081065290891075392210041e-01 2.79145603763543193043261680941214e-01 6.14456089155391627087965389364399e-01 1.55304540973441682766287286199258e-02
            5.03163853150141560099939397332491e-01 3.88410709745710069906721173538244e-01 1.08425437104148369993339429129264e-01 1.71351941586548803253542416769051e-02
            5.03163853150135009784094108908903e-01 1.08425437104147454059344113375118e-01 3.88410709745717563912137393344892e-01 1.71351941586539192885485505257748e-02
            3.88410709745712234841619192593498e-01 5.03163853150140560899217234691605e-01 1.08425437104147204259163572714897e-01 1.71351941586548456308847221407632e-02
            3.88410709745712956486585198945249e-01 1.08425437104149285927334744883410e-01 5.03163853150137674319353209284600e-01 1.71351941586538429607156075462626e-02
            1.08425437104147551203858768076316e-01 5.03163853150138340453167984378524e-01 3.88410709745714122220761055359617e-01 1.71351941586537041828375294016951e-02
            1.08425437104147801004039308736537e-01 3.88410709745717064311776312024449e-01 5.03163853150135231828699033940211e-01 1.71351941586537839801174243348214e-02
            5.18446604434300351016418062499724e-01 2.89296348423914995162675722895074e-01 1.92257047141784653820906214605202e-01 2.08155278830878878726373670815519e-02
            5.18446604434288582652357035840396e-01 1.92257047141778936172329395049019e-01 2.89296348423932481175313569110585e-01 2.08155278830864931549626817286480e-02
            2.89296348423913718406197403965052e-01 5.18446604434304458841609175578924e-01 1.92257047141781822752193420456024e-01 2.08155278830878462392739436381817e-02
            2.89296348423917604186783592012944e-01 1.92257047141783599109032820706489e-01 5.18446604434298796704183587280568e-01 2.08155278830861323324796785527724e-02
            1.92257047141780823551471257815138e-01 5.18446604434278701667437871947186e-01 2.89296348423940474781090870237676e-01 2.08155278830864688688340180533487e-02
            1.92257047141780823551471257815138e-01 2.89296348423907889735318121893215e-01 5.18446604434311231202059389033820e-01 2.08155278830863613159785074913088e-02
            ];
        
    case  24
        
        % ALG. DEG.:   24
        % PTS CARD.:  120
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.2e-16
        
        xyw_bar=[
            4.68044773847062830357534934933028e-03 4.68044773847102295316613407294426e-03 9.90639104523058366957855014334200e-01 3.37914317672483510354708435841076e-04
            4.68044773847029350194448582556106e-03 9.90639104523059477180879639490740e-01 4.68044773847020589840894899680279e-03 3.37914317672430221817930773298144e-04
            9.90639104523057256734830389177660e-01 4.68044773847173939396171249427425e-03 4.68044773847100387120789832806622e-03 3.37914317672531540510949543687502e-04
            3.58515749284085347148298694719415e-02 3.58515749284044130118509485782852e-02 9.28296850143187080028894797578687e-01 2.42500884253115051372717836386528e-03
            3.58515749284033513610836507723434e-02 9.28296850143190410697968673048308e-01 3.58515749284061824297964449215215e-02 2.42500884253095448997439298466361e-03
            9.28296850143188079229616960219573e-01 3.58515749284062726354171957154904e-02 3.58515749284056481349658440649364e-02 2.42500884253109109944812615822229e-03
            4.96163793331061475644361280501471e-01 4.96163793331063862623864224588033e-01 7.67241333787471724292572616832331e-03 4.16222957208837660014744130876352e-03
            4.96163793331063363023503143267590e-01 7.67241333787459928172935974544089e-03 4.96163793331061975244722361821914e-01 4.16222957208834624248661171463937e-03
            7.67241333787459494492066980342315e-03 4.96163793331062530356234674400184e-01 4.96163793331062863423142061947146e-01 4.16222957208834971193356366825356e-03
            8.10973021217381573322313670360018e-02 8.10973021217408635008538908550690e-02 8.37805395756521020800278165552299e-01 7.82875635292406765741723972951149e-03
            8.10973021217391010218022984190611e-02 8.37805395756522242045605253224494e-01 8.10973021217386014214412170986179e-02 7.82875635292391673647482974729428e-03
            8.37805395756521242844883090583608e-01 8.10973021217403500227050017201691e-02 8.10973021217384071324119076962234e-02 7.82875635292404510601205203101927e-03
            1.67282792916414513317846513018594e-01 1.67282792916416844786198225847329e-01 6.65434414167168641895955261134077e-01 1.36181651858831818946438119155573e-02
            1.67282792916414041473061047327064e-01 6.65434414167170640297399586415850e-01 1.67282792916415345985114981885999e-01 1.36181651858830726070648253767104e-02
            6.65434414167167864739838023524499e-01 1.67282792916414596584573359905335e-01 1.67282792916417538675588616570167e-01 1.36181651858830101570196902116550e-02
            4.51971022222129203971263677885872e-01 4.51971022222131368906161696941126e-01 9.60579555557394826337258564308286e-02 1.45966383712733630689806219038474e-02
            4.51971022222132146062278934550704e-01 9.60579555557396630449673580187664e-02 4.51971022222128204770541515244986e-01 1.45966383712732902105946308779494e-02
            9.60579555557397879450576283488772e-02 4.51971022222131646461917853230261e-01 4.51971022222128537837448902791948e-01 1.45966383712731358202052689421180e-02
            2.62525142845062731122851573672961e-01 2.62525142845060122098743704555091e-01 4.74949714309877091267253490514122e-01 2.00618272483955063789640149707338e-02
            2.62525142845060344143348629586399e-01 4.74949714309878590068336734475452e-01 2.62525142845061065788314635938150e-01 2.00618272483955063789640149707338e-02
            4.74949714309880810514385984788532e-01 2.62525142845060788232558479649015e-01 2.62525142845058345741904304304626e-01 2.00618272483954890317292552026629e-02
            3.67318860036336902741282983697602e-01 3.67318860036336625185526827408466e-01 2.65362279927326527584341420151759e-01 2.13423746045388190717417842279247e-02
            3.67318860036338457053517458916758e-01 2.65362279927327138207004963987856e-01 3.67318860036334460250628808353213e-01 2.13423746045388051939539764134679e-02
            2.65362279927327859851970970339607e-01 3.67318860036335959051712052314542e-01 3.67318860036336181096316977345850e-01 2.13423746045387913161661685990111e-02
            9.23455229698260882109650538041024e-01 7.19187624498168059927749595772184e-02 4.62600785192231189757450238175807e-03 1.26054005455723729806072341830259e-03
            9.23455229698260882109650538041024e-01 4.62600785192418626629029532182358e-03 7.19187624498149324914209046255564e-02 1.26054005455754542831814379866273e-03
            7.19187624498129618455521949726972e-02 9.23455229698263657667212100932375e-01 4.62600785192335273166008846601471e-03 1.26054005455745023536739957137343e-03
            7.19187624498160982255967610399239e-02 4.62600785192388355704373736898560e-03 9.23455229698259993931230837915791e-01 1.26054005455757513545766990148422e-03
            4.62600785192390263900197311386364e-03 9.23455229698260882109650538041024e-01 7.19187624498152100471770609146915e-02 1.26054005455754521147770930156184e-03
            4.62600785192329375106190525457350e-03 7.19187624498160982255967610399239e-02 9.23455229698260660065045613009715e-01 1.26054005455738106326879499619054e-03
            9.63258241935609049733102438040078e-01 2.93101710973737335230726586132732e-02 7.43158696701721674382490334664908e-03 1.30769133371010722217964161728787e-03
            9.63258241935611159156849225837504e-01 7.43158696701679954282893092454287e-03 2.93101710973720404329601052495491e-02 1.30769133371003501431495408269257e-03
            2.93101710973702675455676569526986e-02 9.63258241935612713469083701056661e-01 7.43158696701706755760596934123896e-03 1.30769133371002178704844975953847e-03
            2.93101710973716553243484383983741e-02 7.43158696701670066359080024653849e-03 9.63258241935611603246059075900121e-01 1.30769133371002720805931218706064e-03
            7.43158696701625657438095018392232e-03 9.63258241935612602446781238541007e-01 2.93101710973711071517300297273323e-02 1.30769133370994914550289323074139e-03
            7.43158696701714475280065030915466e-03 2.93101710973726579945175529928747e-02 9.63258241935610159956127063196618e-01 1.30769133371013367671265026359606e-03
            8.65062183147061403154509662272176e-01 1.27671748918951943885247146681650e-01 7.26606793398665296024319104617462e-03 2.32188188789964229663542738535398e-03
            8.65062183147064511778978612710489e-01 7.26606793398609871609261645630795e-03 1.27671748918949390372290508821607e-01 2.32188188789954298371642771314782e-03
            1.27671748918951111217978677814244e-01 8.65062183147062846444441674975678e-01 7.26606793398598682642841595225036e-03 2.32188188789963752614586844913447e-03
            1.27671748918952387974456996744266e-01 7.26606793398645606912866767856940e-03 8.65062183147061181109904737240868e-01 2.32188188789959329069723104055356e-03
            7.26606793398613688000908794606403e-03 8.65062183147062735422139212460024e-01 1.27671748918951166729129909072071e-01 2.32188188789952867224775090448929e-03
            7.26606793398584457910338585406862e-03 1.27671748918951111217978677814244e-01 8.65062183147063068489046600006986e-01 2.32188188789945668122349786699488e-03
            7.93149355629101493470045625144849e-01 1.99376854580636947433802674822800e-01 7.47378979026155909615170003235107e-03 3.11613040428292999151071995811435e-03
            7.93149355629101493470045625144849e-01 7.47378979026149404402135090208503e-03 1.99376854580637002944953906080627e-01 3.11613040428300284989671098401232e-03
            1.99376854580639029101973846991314e-01 7.93149355629099384046298837347422e-01 7.47378979026158685172731566126458e-03 3.11613040428296902278892943627397e-03
            1.99376854580637002944953906080627e-01 7.47378979026163889343159496547742e-03 7.93149355629101382447743162629195e-01 3.11613040428293129255332694071967e-03
            7.47378979026150185027699279771696e-03 7.93149355629101493470045625144849e-01 1.99376854580637030700529521709541e-01 3.11613040428291481268030516105227e-03
            7.47378979026160506632381341773907e-03 1.99376854580638029901251684350427e-01 7.93149355629100383247020999988308e-01 3.11613040428299547732193808258216e-03
            7.03167527087426136311876234685769e-01 2.89355487561615498659506329204305e-01 7.47698535095836502861743610992562e-03 3.71782953244761309521959624646570e-03
            7.03167527087424581999641759466613e-01 7.47698535095845870368513885750872e-03 2.89355487561616941949438341907808e-01 3.71782953244765516226388868403774e-03
            2.89355487561616608882530954360845e-01 7.03167527087424915066549147013575e-01 7.47698535095842053976866736775264e-03 3.71782953244760268687874038562313e-03
            2.89355487561615443148355097946478e-01 7.47698535095848038772858856759740e-03 7.03167527087426025289573772170115e-01 3.71782953244765299385954371302887e-03
            7.47698535095843094810952322859521e-03 7.03167527087425692222666384623153e-01 2.89355487561615887237564948009094e-01 3.71782953244764518760390181739695e-03
            7.47698535095859314475452706005854e-03 2.89355487561616941949438341907808e-01 7.03167527087424470977339296950959e-01 3.71782953244769636194644313320623e-03
            8.94987050429267849693815151113085e-01 7.57231791336120396707087820686866e-02 2.92897704371201106354760668182280e-02 3.79678202290759537335107509647969e-03
            8.94987050429267738671512688597431e-01 2.92897704371233684461639512619513e-02 7.57231791336088894128764081870031e-02 3.79678202290746049860081789972810e-03
            7.57231791336077236787005517726357e-02 8.94987050429270736273679176520091e-01 2.92897704371215539254080795217305e-02 3.79678202290756761777545946756618e-03
            7.57231791336120257929209742542298e-02 2.92897704371226676178796566318852e-02 8.94987050429265296180858513253042e-01 3.79678202290733646587228555802085e-03
            2.92897704371223970010174042499784e-02 8.94987050429267849693815151113085e-01 7.57231791336097082023570692399517e-02 3.79678202290736422144790118693436e-03
            2.92897704371209051388280641958772e-02 7.57231791336085285903934050111275e-02 8.94987050429270625251376714004437e-01 3.79678202290747307534601873157953e-03
            6.01693105035377984179945087817032e-01 3.90708026101555361275785571706365e-01 7.59886886306665454426934047660325e-03 4.05968654765336261353025903986236e-03
            6.01693105035377984179945087817032e-01 7.59886886306671439222926167644800e-03 3.90708026101555305764634340448538e-01 4.05968654765332098016683559649209e-03
            3.90708026101554140030458484034170e-01 6.01693105035379094402969712973572e-01 7.59886886306682107772303425008431e-03 4.05968654765332358225204956170273e-03
            3.90708026101555416786936802964192e-01 7.59886886306669877971797788518415e-03 6.01693105035377873157642625301378e-01 4.05968654765338256285023277314394e-03
            7.59886886306669357554754995476287e-03 6.01693105035377984179945087817032e-01 3.90708026101555305764634340448538e-01 4.05968654765335740935983110944107e-03
            7.59886886306676036240137506183601e-03 3.90708026101555416786936802964192e-01 6.01693105035377873157642625301378e-01 4.05968654765334873574245122540560e-03
            8.22015697438519876705242950265529e-01 1.39604852336159324099895684412331e-01 3.83794502253207991948613653221400e-02 6.42042675808840272722033404306785e-03
            8.22015697438525094753458688501269e-01 3.83794502253187383433719048753119e-02 1.39604852336156159964275502716191e-01 6.42042675808830645006741733027411e-03
            1.39604852336156076697548655829451e-01 8.22015697438525094753458688501269e-01 3.83794502253188563045682712981943e-02 6.42042675808830384798220336506347e-03
            1.39604852336156298742153580860759e-01 3.83794502253202440833490527438698e-02 8.22015697438523540441224213282112e-01 6.42042675808828129657701566657124e-03
            3.83794502253187591600536165969970e-02 8.22015697438525094753458688501269e-01 1.39604852336156159964275502716191e-01 6.42042675808822752014926038555132e-03
            3.83794502253179126149973399151349e-02 1.39604852336156465275607274634240e-01 8.22015697438525649864971001079539e-01 6.42042675808821190763797659428747e-03
            7.37522761587901753976836971560260e-01 2.23633835367293892781859199203609e-01 3.88434030448043532413038292361307e-02 7.98338599534551517089564498519394e-03
            7.37522761587902198066046821622876e-01 3.88434030448043601801977331433591e-02 2.23633835367293448692649349140993e-01 7.98338599534571986826581024843108e-03
            2.23633835367292616025380880273588e-01 7.37522761587902642155256671685493e-01 3.88434030448047140637868324120063e-02 7.98338599534572854188319013246655e-03
            2.23633835367291838869263642664009e-01 3.88434030448045336525453308240685e-02 7.37522761587903641355978834326379e-01 7.98338599534571466409538231800980e-03
            3.88434030448042144634257510915631e-02 7.37522761587902753177559134201147e-01 2.23633835367293087870166345965117e-01 7.98338599534572160298928622523817e-03
            3.88434030448049708028612769794563e-02 2.23633835367293809515132352316868e-01 7.37522761587901198865324658981990e-01 7.98338599534582742112132081047093e-03
            6.39399415339909960032116487127496e-01 3.21417139704212873052568966159015e-01 3.91834449558771669153145467134891e-02 8.99182029931184893800644175598791e-03
            6.39399415339909293898301712033572e-01 3.91834449558776665156756280339323e-02 3.21417139704213039586022659932496e-01 8.99182029931190444915767301381493e-03
            3.21417139704212206918754191065091e-01 6.39399415339910071054418949643150e-01 3.91834449558776665156756280339323e-02 8.99182029931199638950189978459093e-03
            3.21417139704212651007964041127707e-01 3.91834449558775346766914537965931e-02 6.39399415339909737987511562096188e-01 8.99182029931209006456960253217403e-03
            3.91834449558775277377975498893647e-02 6.39399415339908627764486936939647e-01 3.21417139704213816742139897542074e-01 8.99182029931202761452446736711863e-03
            3.91834449558780897882037663748633e-02 3.21417139704213483675232509995112e-01 6.39399415339908405719882011908339e-01 8.99182029931198771588451990055546e-03
            5.33728924897748147060383416828699e-01 4.26438949417778823125502185575897e-01 3.98321256844730298141143975954037e-02 9.52049141729484522944648716702432e-03
            5.33728924897748924216500654438278e-01 3.98321256844731755308863796471996e-02 4.26438949417777879435931254192838e-01 9.52049141729477584050744809474054e-03
            4.26438949417778212502838641739800e-01 5.33728924897748591149593266891316e-01 3.98321256844731408364168601110578e-02 9.52049141729488686280991061039458e-03
            4.26438949417779711303921885701129e-01 3.98321256844731616530985718327429e-02 5.33728924897747147859661254187813e-01 9.52049141729489033225686256400877e-03
            3.98321256844731755308863796471996e-02 5.33728924897746814792753866640851e-01 4.26438949417780044370829273248091e-01 9.52049141729482267804129946853209e-03
            3.98321256844733212476583616989956e-02 4.26438949417778212502838641739800e-01 5.33728924897748369104988341860007e-01 9.52049141729477410578397211793344e-03
            7.52006738431794996557755439425819e-01 1.55247740375082515784654901835893e-01 9.27455211931224876575896587382886e-02 1.06091068170761169270344126402961e-02
            7.52006738431800214605971177661559e-01 9.27455211931191847440913988975808e-02 1.55247740375080600649937423440861e-01 1.06091068170760423339249456375910e-02
            1.55247740375080545138786192183034e-01 7.52006738431799548472156402567634e-01 9.27455211931199619002086365071591e-02 1.06091068170760180477962819622917e-02
            1.55247740375080239827454420264985e-01 9.27455211931215439680187273552292e-02 7.52006738431798216204526852379786e-01 1.06091068170762279493368751559501e-02
            9.27455211931191292329401676397538e-02 7.52006738431800103583668715145905e-01 1.55247740375080711672239885956515e-01 1.06091068170760284561371378231343e-02
            9.27455211931186435103668941337673e-02 1.55247740375083848052284452023741e-01 7.52006738431797550070712077285862e-01 1.06091068170762158062725433183004e-02
            6.62157408951895676274546076456318e-01 2.44250018940710467152399587575928e-01 9.35925721073938565730543359677540e-02 1.25098769072565160764831659889751e-02
            6.62157408951895787296848538971972e-01 9.35925721073948280182008829797269e-02 2.44250018940709384684950578048301e-01 1.25098769072566375071264843654717e-02
            2.44250018940708829573438265470031e-01 6.62157408951895787296848538971972e-01 9.35925721073953553741375799290836e-02 1.25098769072566669974255759711923e-02
            2.44250018940709912040887274997658e-01 9.35925721073947169958984204640728e-02 6.62157408951895343207638688909356e-01 1.25098769072568404697731736519017e-02
            9.35925721073943839289910329171107e-02 6.62157408951895787296848538971972e-01 2.44250018940709856529736043739831e-01 1.25098769072566617932551480407710e-02
            9.35925721073961880414060487964889e-02 2.44250018940713631288019769272069e-01 6.62157408951890125159422950673616e-01 1.25098769072568404697731736519017e-02
            5.60254721210861617031184778170427e-01 3.44727467648918373122057801083429e-01 9.50178111402200098467574207461439e-02 1.40002051992718385547975756821870e-02
            5.60254721210862838276511865842622e-01 9.50178111402209257807527365002898e-02 3.44727467648916263698311013286002e-01 1.40002051992717344713890170737614e-02
            3.44727467648916152676008550770348e-01 5.60254721210862838276511865842622e-01 9.50178111402210090474795833870303e-02 1.40002051992718767187140471719431e-02
            3.44727467648918983744721344919526e-01 9.50178111402207314917234270978952e-02 5.60254721210860395785857690498233e-01 1.40002051992720016188043175020539e-02
            9.50178111402206898583600036545249e-02 5.60254721210862061120394628233043e-01 3.44727467648917262899033175926888e-01 1.40002051992717639616881086794820e-02
            9.50178111402216474257187428520410e-02 3.44727467648920482545804588880856e-01 5.60254721210857842272901052638190e-01 1.40002051992717466144533489114110e-02
            5.73642212535126372330296362633817e-01 2.57088626651898688457720254518790e-01 1.69269160812974939211983382847393e-01 1.64576033663149622066068644699044e-02
            5.73642212535122597572012637101579e-01 1.69269160812976965369003323758079e-01 2.57088626651900464814559654769255e-01 1.64576033663148026120470746036517e-02
            2.57088626651898188857359173198347e-01 5.73642212535124262906549574836390e-01 1.69269160812977603747242483223090e-01 1.64576033663150662900154230783301e-02
            2.57088626651899243569232567097060e-01 1.69269160812976410257491011179809e-01 5.73642212535124373928852037352044e-01 1.64576033663150766983562789391726e-02
            1.69269160812976271479612933035241e-01 5.73642212535124151884247112320736e-01 2.57088626651899576636139954644023e-01 1.64576033663149969010763840060463e-02
            1.69269160812978797236993955266371e-01 2.57088626651900686859164579800563e-01 5.73642212535120488148265849304153e-01 1.64576033663149934316294320524321e-02
            4.68278059755925679663590699419728e-01 3.59525370113814635164573019210366e-01 1.72196570130259629660685050112079e-01 1.84067508755945546949917002166330e-02
            4.68278059755924569440566074263188e-01 1.72196570130260739883709675268619e-01 3.59525370113814635164573019210366e-01 1.84067508755943326503867751853250e-02
            3.59525370113814135564211937889922e-01 4.68278059755925291085532080614939e-01 1.72196570130260573350255981495138e-01 1.84067508755945442866508443557905e-02
            3.59525370113814912720329175499501e-01 1.72196570130260406816802287721657e-01 4.68278059755924680462868536778842e-01 1.84067508755944714282648533298925e-02
            1.72196570130260601105831597124052e-01 4.68278059755924958018624693067977e-01 3.59525370113814468631119325436885e-01 1.84067508755943951004319103503803e-02
            1.72196570130261378261948834733630e-01 3.59525370113815190276085331788636e-01 4.68278059755923459217541449106648e-01 1.84067508755943534670684869070101e-02
            ];
        
    case  25
        
        % ALG. DEG.:   25
        % PTS CARD.:  136
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.4e-16
        
        xyw_bar=[
            3.33333333333332593184650249895640e-01 3.33333333333333481363069950020872e-01 3.33333333333333925452279800083488e-01 1.91335092595674785553860886011535e-02
            5.71433885823402661074954522746339e-03 5.71433885823497897393785649455822e-03 9.88571322283530951047225698857801e-01 4.03376930733677190044073057606511e-04
            5.71433885823444728119246960318378e-03 9.88571322283530284913410923763877e-01 5.71433885823524612135315692285076e-03 4.03376930733716275532391160041357e-04
            9.88571322283531395136435548920417e-01 5.71433885823455309932450418841654e-03 5.71433885823405176423994689116626e-03 4.03376930733648512896610816014231e-04
            2.72494286004120213340318201744594e-02 2.72494286004143874968530525393362e-02 9.45501142799173632802478550729575e-01 2.04894605750170419322664372430154e-03
            2.72494286004084998453755872560578e-02 9.45501142799175964270830263558310e-01 2.72494286004155705782636687217746e-02 2.04894605750174886235615012708422e-03
            9.45501142799175964270830263558310e-01 2.72494286004124376676660546081621e-02 2.72494286004115980615036818335284e-02 2.04894605750160097717982310427942e-03
            7.66556657710398547944663505404606e-02 7.66556657710376621039927158562932e-02 8.46688668457922455345965317974333e-01 6.05556013297073661427738500151463e-03
            7.66556657710362188140607031527907e-02 8.46688668457924675792014568287414e-01 7.66556657710391053939247285597958e-02 6.05556013297058135652628507727968e-03
            8.46688668457927229304971206147457e-01 7.66556657710359690138801624925691e-02 7.66556657710368016811486313599744e-02 6.05556013297066982742355989444150e-03
            4.82135684364817795888313867180841e-01 4.82135684364816408109533085735166e-01 3.57286312703657960021530470839934e-02 7.72624379420794923228887896016204e-03
            4.82135684364817573843708942149533e-01 3.57286312703658168188347588056786e-02 4.82135684364816630154138010766474e-01 7.72624379420830398323971621721284e-03
            3.57286312703654906908212751659448e-02 4.82135684364817795888313867180841e-01 4.82135684364816685665289242024301e-01 7.72624379420842888332998654732364e-03
            1.49523257795367153777377211554267e-01 1.49523257795364461486542495549656e-01 7.00953484409268412491655908524990e-01 1.11579277905625544320411535181847e-02
            1.49523257795361963484737088947440e-01 7.00953484409271410093822396447649e-01 1.49523257795366570910289283347083e-01 1.11579277905625631056585334022202e-02
            7.00953484409271299071519933931995e-01 1.49523257795363379019093486022030e-01 1.49523257795365321909386580045975e-01 1.11579277905624885125490663995151e-02
            4.22530364706221017723208888128283e-01 4.22530364706225070037248769949656e-01 1.54939270587553856728391110664234e-01 1.56405465194085074864815965156595e-02
            4.22530364706225347593004926238791e-01 1.54939270587553634683786185632925e-01 4.22530364706221073234360119386110e-01 1.56405465194085803448675875415574e-02
            1.54939270587552690994215254249866e-01 4.22530364706223404702711832214845e-01 4.22530364706223848791921682277462e-01 1.56405465194084346280956054897615e-02
            2.35017663942219401373989740022807e-01 2.35017663942217097661213642822986e-01 5.29964672115563417698069770267466e-01 1.62235276308906918585339695937364e-02
            2.35017663942217486239272261627775e-01 5.29964672115563639742674695298774e-01 2.35017663942218901773628658702364e-01 1.62235276308904420583534289335148e-02
            5.29964672115565527121816558064893e-01 2.35017663942220955686224215241964e-01 2.35017663942213517191959226693143e-01 1.62235276308906432862766422431378e-02
            9.68738481181608435122143418993801e-01 2.64387673203299901347751443836387e-02 4.82275149806157474308143662256043e-03 6.55986048218150721855757012690447e-04
            9.68738481181608435122143418993801e-01 4.82275149806082274045460067668500e-03 2.64387673203307430047637183179177e-02 6.55986048218209702453940224131657e-04
            2.64387673203289839951590778355239e-02 9.68738481181608435122143418993801e-01 4.82275149806254965767493558814749e-03 6.55986048218261527317785031243602e-04
            2.64387673203308991298765562305562e-02 4.82275149806150448678065956187311e-03 9.68738481181607546943723718868569e-01 6.55986048218284946084710718139377e-04
            4.82275149805882780845722734852643e-03 9.68738481181608435122143418993801e-01 2.64387673203327899784653709502891e-02 6.55986048217894958563267682194464e-04
            4.82275149806148627218416180539862e-03 2.64387673203308991298765562305562e-02 9.68738481181607657966026181384223e-01 6.55986048218139337732945914893889e-04
            9.35914465591766275842644517979352e-01 5.82420655518004273809573589915090e-02 5.84346885643329677639812302913924e-03 1.18175318106926549761859224219052e-03
            9.35914465591761723928243554837536e-01 5.84346885643438184593234652197680e-03 5.82420655518038968279093126056978e-02 1.18175318106940601022014636356516e-03
            5.82420655518003371753366081975400e-02 9.35914465591765942775737130432390e-01 5.84346885643372004892626137007028e-03 1.18175318106939885448580795923590e-03
            5.82420655518037094777739071105316e-02 5.84346885643469496351976033565734e-03 9.35914465591761612905941092321882e-01 1.18175318106944720990270081273366e-03
            5.84346885643483113931262451501425e-03 9.35914465591766275842644517979352e-01 5.82420655517988938854045954940375e-02 1.18175318106929173531116639139782e-03
            5.84346885643377469271575463949375e-03 5.82420655518003371753366081975400e-02 9.35914465591765942775737130432390e-01 1.18175318106931406987591959278916e-03
            8.85737420571855227358071260823635e-01 1.07957673880294022228376604743971e-01 6.30490554785075041355213443239336e-03 1.93167760124472362957825932738842e-03
            8.85737420571854561224256485729711e-01 6.30490554785023606804150730908987e-03 1.07957673880295201840340268972795e-01 1.93167760124455254247544111478874e-03
            1.07957673880294591217676725136698e-01 8.85737420571854339179651560698403e-01 6.30490554785112511382294542272575e-03 1.93167760124480039109207130110235e-03
            1.07957673880294896529008497054747e-01 6.30490554785039566260129717534255e-03 8.85737420571854783268861410761019e-01 1.93167760124461304095666580593615e-03
            6.30490554784987871500545608682842e-03 8.85737420571860889495496849121992e-01 1.07957673880289206636007293127477e-01 1.93167760124455492772022058289849e-03
            6.30490554785099154011529520857948e-03 1.07957673880293189561108135876566e-01 8.85737420571855782469583573401906e-01 1.93167760124478738066600147504914e-03
            8.16939712186552924322313629090786e-01 1.76404098155251098933860021134024e-01 6.65618965819597674382634977519047e-03 2.60257074853620547216781488941706e-03
            8.16939712186550592853961916262051e-01 6.65618965819580327147875209448102e-03 1.76404098155253596935665427736240e-01 2.60257074853620373744433891260996e-03
            1.76404098155251293222889330536418e-01 8.16939712186552813300011166575132e-01 6.65618965819592123267511851736344e-03 2.60257074853618769125218612714434e-03
            1.76404098155253291624333655818191e-01 6.65618965819602878553062907940330e-03 8.16939712186550703876264378777705e-01 2.60257074853628266736249585733276e-03
            6.65618965819592383476033248257409e-03 8.16939712186552924322313629090786e-01 1.76404098155251154445011252391851e-01 2.60257074853633601010938214415091e-03
            6.65618965819600363204022741570043e-03 1.76404098155251293222889330536418e-01 8.16939712186552702277708704059478e-01 2.60257074853627139165990200808665e-03
            7.34078265782891925539388466859236e-01 2.59206444374593891755154118072824e-01 6.71528984251418270545741506794002e-03 3.03986398041122991842133416184879e-03
            7.34078265782887817714197353780037e-01 6.71528984251453138487608640616600e-03 2.59206444374597666513437843605061e-01 3.03986398041141076334370474398838e-03
            2.59206444374594335844363968135440e-01 7.34078265782891259405573691765312e-01 6.71528984251446026121357135707512e-03 3.03986398041133139974467880506381e-03
            2.59206444374597722024589074862888e-01 6.71528984251433969793199096898206e-03 7.34078265782887928736499816295691e-01 3.03986398041136696157593632960925e-03
            6.71528984251449061887440095119928e-03 7.34078265782889372026431828999193e-01 2.59206444374596167712354599643731e-01 3.03986398041136912998028130061812e-03
            6.71528984251436051461370269066720e-03 2.59206444374595501578539824549807e-01 7.34078265782890149182549066608772e-01 3.03986398041136912998028130061812e-03
            6.41753644337056394775231638050172e-01 3.51505881363440575437806501213345e-01 6.74047429950302978696186073648278e-03 3.28693081105050580875692745053129e-03
            6.41753644337053841262275000190130e-01 6.74047429950344525323435718178189e-03 3.51505881363442740372704520268599e-01 3.28693081105059688173941623290375e-03
            3.51505881363440519926655269955518e-01 6.41753644337056172730626713018864e-01 6.74047429950330734271801702561788e-03 3.28693081105063721406023269366869e-03
            3.51505881363444683262997614292544e-01 6.74047429950306881824007021464240e-03 6.41753644337052286950040524970973e-01 3.28693081105051708445952129977741e-03
            6.74047429950314601343475118255810e-03 6.41753644337056394775231638050172e-01 3.51505881363440408904352807439864e-01 3.28693081105043338405180541883510e-03
            6.74047429950329346493020921116113e-03 3.51505881363442962417309445299907e-01 6.41753644337053730239972537674475e-01 3.28693081105065152552890950232722e-03
            5.45201712627926382559451212728163e-01 4.47959453755782277539765345863998e-01 6.83883361629133990078344140783884e-03 3.40174363689335883947917338332445e-03
            5.45201712627927492782475837884704e-01 6.83883361629142316751028829457937e-03 4.47959453755781056294438258191803e-01 3.40174363689330679777489407911162e-03
            4.47959453755781500383648108254420e-01 5.45201712627927159715568450337742e-01 6.83883361629133990078344140783884e-03 3.40174363689340350860867978610713e-03
            4.47959453755785386164234296302311e-01 6.83883361629121500069317107772804e-03 5.45201712627923384957284724805504e-01 3.40174363689327730747580247339101e-03
            6.83883361629120372499057722848192e-03 5.45201712627929824250827550713439e-01 4.47959453755778946870691470394377e-01 3.40174363689325692447495974590765e-03
            6.83883361629130000214349394127566e-03 4.47959453755785386164234296302311e-01 5.45201712627923273934982262289850e-01 3.40174363689339570235303789047521e-03
            8.99672974696369753822011716692941e-01 6.92285554477007208928895920507784e-02 3.10984698559295252850986912562803e-02 3.59560620720565226465792463272919e-03
            8.99672974696373417557992979709525e-01 3.10984698559305557108434214796944e-02 6.92285554476960302006105507643952e-02 3.59560620720527886542972062500212e-03
            6.92285554476950171221005803090520e-02 8.99672974696373861647202829772141e-01 3.10984698559311212306965899188071e-02 3.59560620720571297997958382097750e-03
            6.92285554476976955351474884992058e-02 3.10984698559316555255271907753922e-02 8.99672974696370642000431416818174e-01 3.59560620720546621556512612016832e-03
            3.10984698559315132782021606772105e-02 8.99672974696363092483863965753699e-01 6.92285554477053421962295942648780e-02 3.59560620720561887123101207919262e-03
            3.10984698559315861365881517031085e-02 6.92285554477025388830924157446134e-02 8.99672974696365868041425528645050e-01 3.59560620720572642408652264123248e-03
            8.36754745906223784679411892284406e-01 1.29450436723390405191480567737017e-01 3.37948173703858101291075399785768e-02 5.20296994687292468223516195280354e-03
            8.36754745906235442021170456428081e-01 3.37948173703829651826069380149420e-02 1.29450436723381578918434797742520e-01 5.20296994687244156174710241202774e-03
            1.29450436723384493253874438778439e-01 8.36754745906229335794535018067108e-01 3.37948173703861431960149275255390e-02 5.20296994687296544823684740777026e-03
            1.29450436723384493253874438778439e-01 3.37948173703840337722681397281121e-02 8.36754745906231445218281805864535e-01 5.20296994687263671813814980282586e-03
            3.37948173703825766045483192101528e-02 8.36754745906224228768621742347023e-01 1.29450436723393180749042130628368e-01 5.20296994687248492983400183220510e-03
            3.37948173703861778904844470616808e-02 1.29450436723389350479607173838303e-01 8.36754745906224561835529129893985e-01 5.20296994687267401469288330417839e-03
            7.60018820808681438983001044107368e-01 2.05092818400494147912382914000773e-01 3.48883607908244131046160418918589e-02 6.39596396083737424309578756265182e-03
            7.60018820808689321566475882718805e-01 3.48883607908248641327197958617035e-02 2.05092818400485821239698225326720e-01 6.39596396083767608498060752708625e-03
            2.05092818400489540486830719601130e-01 7.60018820808685990897402007249184e-01 3.48883607908244686157672731496859e-02 6.39596396083763878842587402573372e-03
            2.05092818400489040886469638280687e-01 3.48883607908252943441418381098629e-02 7.60018820808685768852797082217876e-01 6.39596396083765179885194385178693e-03
            3.48883607908255163887467631411710e-02 7.60018820808677997291624706122093e-01 2.05092818400496534891885858087335e-01 6.39596396083744883620525456535688e-03
            3.48883607908248988271893153978453e-02 2.05092818400489096397620869538514e-01 7.60018820808685990897402007249184e-01 6.39596396083720684228035580076721e-03
            6.72878720973754451684101240971358e-01 2.92117203365808597581576577795204e-01 3.50040756604369507343221812334377e-02 7.14527129784467592010788195011628e-03
            6.72878720973758337464687429019250e-01 3.50040756604392336304165667115740e-02 2.92117203365802435843789908176404e-01 7.14527129784526485872797607612483e-03
            2.92117203365807209802795796349528e-01 6.72878720973754229639496315940050e-01 3.50040756604385050465566564525943e-02 7.14527129784523623579062245880777e-03
            2.92117203365804434245234233458177e-01 3.50040756604377209515455149357877e-02 6.72878720973757893375477578956634e-01 7.14527129784485893343459750326474e-03
            3.50040756604382483074822118851444e-02 6.72878720973754451684101240971358e-01 2.92117203365807265313947027607355e-01 7.14527129784454321376196972437356e-03
            3.50040756604383454519968665863416e-02 2.92117203365801103576160357988556e-01 6.72878720973760557910736679332331e-01 7.14527129784488148483978520175697e-03
            5.78766335729510972285538628057111e-01 3.85847444474737621966653478011722e-01 3.53862197957514057478078939311672e-02 7.58299009147856556412037321024400e-03
            5.78766335729513636820797728432808e-01 3.53862197957528004654825792840711e-02 3.85847444474733569652613596190349e-01 7.58299009147902786792672102933466e-03
            3.85847444474738177078165790589992e-01 5.78766335729509084906396765290992e-01 3.53862197957527380154374441190157e-02 7.58299009147886827336693116308197e-03
            3.85847444474734901920243146378198e-01 3.53862197957514057478078939311672e-02 5.78766335729513747843100190948462e-01 7.58299009147876072051142060104212e-03
            3.53862197957515028923225486323645e-02 5.78766335729514747043822353589348e-01 3.85847444474733736186067289963830e-01 7.58299009147859592178120280436815e-03
            3.53862197957522592317580745202577e-02 3.85847444474732015340379120971193e-01 5.78766335729515635222242053714581e-01 7.58299009147907730754578636833685e-03
            7.76386479467535695064839273982216e-01 1.41173270259467076304105148665258e-01 8.24402502729972286310555773525266e-02 8.44269504963643217798630047354891e-03
            7.76386479467535695064839273982216e-01 8.24402502729935232617108908925729e-02 1.41173270259470795551237642939668e-01 8.44269504963672014208331262352658e-03
            1.41173270259466576703744067344815e-01 7.76386479467535695064839273982216e-01 8.24402502729977282314166586729698e-02 8.44269504963635585015335749403675e-03
            1.41173270259473793153404130862327e-01 8.24402502729952163518234442562971e-02 7.76386479467531032128135848324746e-01 8.44269504963670106012507687864854e-03
            8.24402502729926489610789985817974e-02 7.76386479467535695064839273982216e-01 1.41173270259471683729657343064900e-01 8.44269504963649809747838759221850e-03
            8.24402502729979641538093915187346e-02 1.41173270259467881215798001903750e-01 7.76386479467534140752604798763059e-01 8.44269504963633676819512174915872e-03
            6.95281183841358396691134657885414e-01 2.20694701488690830926131525302480e-01 8.40241146699507723827338168121059e-02 1.00443207868060394027942550110311e-02
            6.95281183841349070817727806570474e-01 8.40241146699540614184442688383569e-02 2.20694701488696853886040116776712e-01 1.00443207868065095128562447257536e-02
            2.20694701488689193347170203196583e-01 6.95281183841358507713437120401068e-01 8.40241146699522989393926764023490e-02 1.00443207868061729765019052251773e-02
            2.20694701488698713509606363913917e-01 8.40241146699534091624173015588894e-02 6.95281183841347849572400718898280e-01 1.00443207868061296084150058050000e-02
            8.40241146699542557074735782407515e-02 6.95281183841349070817727806570474e-01 2.20694701488696631841435191745404e-01 1.00443207868062666515696079727604e-02
            8.40241146699528540509049889806192e-02 2.20694701488689720703106900145940e-01 6.95281183841357397490412495244527e-01 1.00443207868060793014342024775942e-02
            6.04556578856266901844662697840249e-01 3.10779394840663303334338252170710e-01 8.46640263030697948209990499890409e-02 1.12068990501475312937529338341847e-02
            6.04556578856259130283490321744466e-01 8.46640263030742495908853584296594e-02 3.10779394840666634003412127640331e-01 1.12068990501473300658297205245617e-02
            3.10779394840659584087205757896299e-01 6.04556578856267012866965160355903e-01 8.46640263030733475346778504899703e-02 1.12068990501475347631998857877988e-02
            3.10779394840667078092621977702947e-01 8.46640263030706413660553266709030e-02 6.04556578856262238907959272182779e-01 1.12068990501468998544076782764023e-02
            8.46640263030713630110213330226543e-02 6.04556578856262349930261734698433e-01 3.10779394840666300936504740093369e-01 1.12068990501471444504177910062026e-02
            8.46640263030726120119240363237623e-02 3.10779394840659306531449601607164e-01 6.04556578856268123089989785512444e-01 1.12068990501474289450678512025661e-02
            5.06851226677805466458437422261341e-01 4.07374461989679470530489879820379e-01 8.57743113325150630110726979182800e-02 1.18967987907211067422075245758606e-02
            5.06851226677805355436134959745687e-01 8.57743113325167561011852512820042e-02 4.07374461989677860707104173343396e-01 1.18967987907206852044028622117366e-02
            4.07374461989673308792703210201580e-01 5.06851226677810018372838385403156e-01 8.57743113325166728344584043952636e-02 1.18967987907210043935224419442420e-02
            4.07374461989678027240557867116877e-01 8.57743113325149242331946197737125e-02 5.06851226677807131792974359996151e-01 1.18967987907205724473769237192755e-02
            8.57743113325147021885896947424044e-02 5.06851226677805466458437422261341e-01 4.07374461989679859108548498625169e-01 1.18967987907208864323260755213596e-02
            8.57743113325157430226752808266610e-02 4.07374461989673586348459366490715e-01 5.06851226677810684506653160497081e-01 1.18967987907210148018632978050846e-02
            6.17605377498659180446338723413646e-01 2.30766928655942393788791378028691e-01 1.51627693845398425764869898557663e-01 1.32352995690810000539716639877952e-02
            6.17605377498654406487332835240522e-01 1.51627693845405031591866418239078e-01 2.30766928655940561920800746520399e-01 1.32352995690807728051963110260658e-02
            2.30766928655939534964502968250599e-01 6.17605377498657959201011635741452e-01 1.51627693845402533590061011636863e-01 1.32352995690811006679332706426067e-02
            2.30766928655944475456962550197204e-01 1.51627693845402283789880470976641e-01 6.17605377498653185242005747568328e-01 1.32352995690807797440902149332942e-02
            1.51627693845403532790783174277749e-01 6.17605377498652519108190972474404e-01 2.30766928655943948101025853247847e-01 1.32352995690807190287685557450459e-02
            1.51627693845403033190422092957306e-01 2.30766928655941672143825371676940e-01 6.17605377498655183643450072850101e-01 1.32352995690810451567820393847796e-02
            5.23270463532956031649234773794888e-01 3.23002638196493885303084425686393e-01 1.53726898270550083047680800518719e-01 1.49764780310988367129088061346920e-02
            5.23270463532956031649234773794888e-01 1.53726898270555273340320923125546e-01 3.23002638196488667254868687450653e-01 1.49764780310986025252395492657342e-02
            3.23002638196489277877532231286750e-01 5.23270463532956031649234773794888e-01 1.53726898270554745984384226176189e-01 1.49764780310987048739246318973528e-02
            3.23002638196494051836538119459874e-01 1.53726898270550610403617497468076e-01 5.23270463532955254493117536185309e-01 1.49764780310986701794551123612109e-02
            1.53726898270550860203798038128298e-01 5.23270463532956031649234773794888e-01 3.23002638196493108146967188076815e-01 1.49764780310984533390206152603241e-02
            1.53726898270553441472330291617254e-01 3.23002638196491276278976556568523e-01 5.23270463532955254493117536185309e-01 1.49764780310986805877959682220535e-02
            4.31308082067836551232176134362817e-01 3.30276680517266163494838338010595e-01 2.38415237414897285272985527626588e-01 1.80238652394471010609322547679767e-02
            4.31308082067838161055561840839800e-01 2.38415237414900393897454478064901e-01 3.30276680517261500558134912353125e-01 1.80238652394473196360902278456706e-02
            3.30276680517262999359218156314455e-01 4.31308082067836440209873671847163e-01 2.38415237414900560430908171838382e-01 1.80238652394469830997358883450943e-02
            3.30276680517265996961384644237114e-01 2.38415237414897340784136758884415e-01 4.31308082067836662254478596878471e-01 1.80238652394473473916658434745841e-02
            2.38415237414896674650321983790491e-01 4.31308082067837994522108147066319e-01 3.30276680517265330827569869143190e-01 1.80238652394471392248487262577328e-02
            2.38415237414898950607522465361399e-01 3.30276680517264054071091550213168e-01 4.31308082067837050832537215683260e-01 1.80238652394472120832347172836307e-02
            ];
        
    case  27
        
        % ALG. DEG.:   27
        % PTS CARD.:  153
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.0e-16
        
        xyw_bar=[
            6.02872510688659112715326315878883e-03 6.02872510688349898255733450014304e-03 9.87942549786229973207696275494527e-01 4.43235614294061413230563228893288e-04
            6.02872510688389449950985721216057e-03 9.87942549786234525122097238636343e-01 6.02872510688157170477552426746115e-03 4.43235614293717287461016329785934e-04
            9.87942549786236745568146488949424e-01 6.02872510688300979053710904054242e-03 6.02872510688024464131640201003393e-03 4.43235614293479630344807507214000e-04
            2.25804480641049951095133963008266e-02 2.25804480640832486160185510470910e-02 9.54839103871811811785619283909909e-01 1.70838802256757077945026335186185e-03
            2.25804480641088878289934882559464e-02 9.54839103871812144852526671456872e-01 2.25804480640789950740554559160955e-02 1.70838802256684718292034652620259e-03
            9.54839103871770289444498303055298e-01 2.25804480641217698855260920254295e-02 2.25804480641079406699756049192729e-02 1.70838802256833861142881758610201e-03
            4.96948162575856389189254969096510e-01 4.96948162575845342470159948788933e-01 6.10367484829826834058508211455774e-03 2.93659414160908106838920872405652e-03
            4.96948162575851670741400312181213e-01 6.10367484829858492761944788185247e-03 4.96948162575849783362258449415094e-01 2.93659414161008070279224035914467e-03
            6.10367484829798297857328392979070e-03 4.96948162575852780964424937337753e-01 4.96948162575849228250746136836824e-01 2.93659414160977192201351648748187e-03
            6.97453864826078351324412096801098e-02 6.97453864825942071448139358835761e-02 8.60509227034797929967169238807401e-01 4.52841537864233974891581269162089e-03
            6.97453864825669234139837726615951e-02 8.60509227034811585710372128232848e-01 6.97453864826215186312197147344705e-02 4.52841537864415426967168443184164e-03
            8.60509227034811585710372128232848e-01 6.97453864825740427191291814779106e-02 6.97453864826143715704986902892415e-02 4.52841537864409355435002524359334e-03
            1.33222990598758694869019336692872e-01 1.33222990598769824854841203887190e-01 7.33554018802471397009412612533197e-01 9.26039382719270366683961270837244e-03
            1.33222990598766410919040481530828e-01 7.33554018802442087121562508400530e-01 1.33222990598791501959397010068642e-01 9.26039382719397348442402773116555e-03
            7.33554018802459406600746660842560e-01 1.33222990598787810467840131423145e-01 1.33222990598752782931413207734295e-01 9.26039382719229427209928218189816e-03
            4.61535910661840520496923545579193e-01 4.61535910661833803647624563382124e-01 7.69281786763256758554518910386832e-02 1.03250177142071037500459240732198e-02
            4.61535910661845460989383127525798e-01 7.69281786763297420472795806745125e-02 4.61535910661824755329973868356319e-01 1.03250177142078913145040175436407e-02
            7.69281786763209435298094263089297e-02 4.61535910661838633117781682813074e-01 4.61535910661840409474621083063539e-01 1.03250177142063352675460663476770e-02
            2.11399010110189911060274425835814e-01 2.11399010110171869936124267042032e-01 5.77201979779638163492450075864326e-01 1.31536544265109245355205302985269e-02
            2.11399010110194796041582776524592e-01 5.77201979779594975816792157274904e-01 2.11399010110210228141625066200504e-01 1.31536544265102237072362356684607e-02
            5.77201979779581431095891730365111e-01 2.11399010110200957779369446143392e-01 2.11399010110217611124738823491498e-01 1.31536544265149629717726043054427e-02
            3.92179397457324996345562340138713e-01 3.92179397457366463175532089735498e-01 2.15641205085308540478905570125789e-01 1.58746081641986842847291683256117e-02
            3.92179397457315004338340713729849e-01 2.15641205085320392109693443671858e-01 3.92179397457364631307541458227206e-01 1.58746081641979938647857295563881e-02
            2.15641205085286835618774148315424e-01 3.92179397457343759114678505284246e-01 3.92179397457369405266547346400330e-01 1.58746081641992324573475769966535e-02
            3.02074000493948113810205313711776e-01 3.02074000493992356197736626199912e-01 3.95851999012059474480906828830484e-01 1.68896031376139753310550162268555e-02
            3.02074000493948557899415163774393e-01 3.95851999012071742445328936810256e-01 3.02074000493979644144104668157524e-01 1.68896031376141002311452865569663e-02
            3.95851999012003130662407102136058e-01 3.02074000494003291894529183991835e-01 3.02074000493993632954214945129934e-01 1.68896031376139718616080642732413e-02
            9.72577577219754130233297928498359e-01 2.56411289038486409075368044341303e-02 1.78129387639722885916526706751029e-03 3.01712733455144642338158211813948e-04
            9.72577577219754130233297928498359e-01 1.78129387641991015177322932316883e-03 2.56411289038259611328118836581780e-02 3.01712733456077815148016485480298e-04
            2.56411289038599408962593173555433e-02 9.72577577219763789173612167360261e-01 1.78129387637626646068156333058141e-03 3.01712733454015228935080061845042e-04
            2.56411289038599408962593173555433e-02 1.78129387637662663264326301515439e-03 9.72577577219763456106704779813299e-01 3.01712733453783534930819909547495e-04
            1.78129387640009765811366371224267e-03 9.72577577219754130233297928498359e-01 2.56411289038457335109910673054401e-02 3.01712733455220699120558069949993e-04
            1.78129387638326411834321927329938e-03 2.56411289038599408962593173555433e-02 9.72577577219756794768557028874056e-01 3.01712733454599180225180754533199e-04
            9.44757428656674114186841961782193e-01 4.89478566028413442090538865159033e-02 6.29471474048454160410415170190390e-03 1.03404277559088144007592990902822e-03
            9.44757428656674114186841961782193e-01 6.29471474046813285474488708359786e-03 4.89478566028577546931366271110164e-02 1.03404277559008671988349803427809e-03
            4.89478566028448414115814557590056e-02 9.44757428656667563870996673358604e-01 6.29471474048759471742187088239007e-03 1.03404277559139209929917058161664e-03
            4.89478566028448414115814557590056e-02 6.29471474048211993013168807920010e-03 9.44757428656673003963817336625652e-01 1.03404277559136477740442394690490e-03
            6.29471474048273402224218386891152e-03 9.44757428656674114186841961782193e-01 4.89478566028431760770445180241950e-02 1.03404277559125483930413391675529e-03
            6.29471474049513469301020407442593e-03 4.89478566028448414115814557590056e-02 9.44757428656660014354429222294129e-01 1.03404277559268641985268377680995e-03
            9.00140112535737002552593821746996e-01 9.43702253266477247839461028888763e-02 5.48966213761527266346007536412799e-03 1.59319950222422358418472576602198e-03
            9.00140112535686154338065989577444e-01 5.48966213761783138058714115459225e-03 9.43702253266960194855172971983848e-02 1.59319950222490511367035015410920e-03
            9.43702253266545387777597397871432e-02 9.00140112535729897125236220745137e-01 5.48966213761559185257965509663336e-03 1.59319950222424982187729991522929e-03
            9.43702253266842372436684627245995e-02 5.48966213761628400724656984266403e-03 9.00140112535699477014361491455929e-01 1.59319950222497211736460975828322e-03
            5.48966213761158811079710062585946e-03 9.00140112535713687869076693459647e-01 9.43702253266747170812323020072654e-02 1.59319950222375455832490853680383e-03
            5.48966213761193679021577196408543e-03 9.43702253266681945209626292125904e-02 9.00140112535719905118014594336273e-01 1.59319950222398592706851694345005e-03
            8.35926640580037294547821602463955e-01 1.58178278042184511376078148714441e-01 5.89508137777819407610024882160360e-03 2.10802503773852142540667919945463e-03
            8.35926640580002766611755760095548e-01 5.89508137778006844481604176166911e-03 1.58178278042217151933002128316730e-01 2.10802503773838004544338708967643e-03
            1.58178278042176712059330156989745e-01 8.35926640580041735439920103090117e-01 5.89508137778149698959850866231136e-03 2.10802503773900324485213175762510e-03
            1.58178278042216791110519125140854e-01 5.89508137777945955687597390237897e-03 8.35926640580003765812477922736434e-01 2.10802503773829634503567120873413e-03
            5.89508137778011354762641715865357e-03 8.35926640580004987057805010408629e-01 1.58178278042214848220226031116908e-01 2.10802503773880288429065643640570e-03
            5.89508137777934333040308345630365e-03 1.58178278042213321663567171526665e-01 8.35926640580007318526156723237364e-01 2.10802503773857216606835152106214e-03
            7.61488498658509094241253478685394e-01 2.32483150304506530625658911048959e-01 6.02835103698437513308761026564753e-03 2.42083798008871625001559024781272e-03
            7.61488498658509094241253478685394e-01 6.02835103698335251359852193786537e-03 2.32483150304507557581956689318758e-01 2.42083798008723609720971303715942e-03
            2.32483150304480662429185144901567e-01 7.61488498658535295504634632379748e-01 6.02835103698406982175583834759891e-03 2.42083798008831249312655664596150e-03
            2.32483150304507835137712845607894e-01 6.02835103698241836500670842724503e-03 7.61488498658509760375068253779318e-01 2.42083798008681672780939564404434e-03
            6.02835103698410798567230983735499e-03 7.61488498658503099036920502840076e-01 2.32483150304512831141323658812325e-01 2.42083798008762077214051089413260e-03
            6.02835103698397788141161157682291e-03 2.32483150304507807382137229978980e-01 7.61488498658508206062833778560162e-01 2.42083798008738745183299201357841e-03
            6.79274857925002639724709752044873e-01 3.14735969095647993842845835388289e-01 5.98917297934936643244441256683785e-03 2.63574074337429064066040673708358e-03
            6.79274857925028618943485980707919e-01 5.98917297934907846834740041686018e-03 3.14735969095622292179825763014378e-01 2.63574074337370907461508551250517e-03
            3.14735969095626788583075494898367e-01 6.79274857925025066229807180206990e-01 5.98917297934808967596609363681637e-03 2.63574074337424250208394838068671e-03
            3.14735969095621681557162219178281e-01 5.98917297934821457605636396692717e-03 6.79274857925030062233417993411422e-01 2.63574074337336473200510411629693e-03
            5.98917297934755538113549278023129e-03 6.79274857925024622140597330144374e-01 3.14735969095627843294948888797080e-01 2.63574074337298396020212720713971e-03
            5.98917297934768461803445305235982e-03 3.14735969095620127244927743959124e-01 6.79274857925032282679467243724503e-01 2.63574074337311753390977742128598e-03
            5.90173779527676689582449398585595e-01 4.03765902224775929951761099800933e-01 6.06031824754738046578950161347166e-03 2.84708688341651092526540800520252e-03
            5.90173779527698449953732051653787e-01 6.06031824754737352689559770624328e-03 4.03765902224754169580478446732741e-01 2.84708688341714366565327054559020e-03
            4.03765902224775929951761099800933e-01 5.90173779527677799805474023742136e-01 6.06031824754627024276487645693123e-03 2.84708688341669610699646852935984e-03
            4.03765902224754891225444453084492e-01 6.06031824754691989670662977118809e-03 5.90173779527698227909127126622479e-01 2.84708688341701139298822731404925e-03
            6.06031824754597100296527045770745e-03 5.90173779527697783819917276559863e-01 4.03765902224756279004225234530168e-01 2.84708688341637605051515080845093e-03
            6.06031824754629279417006415542346e-03 4.03765902224754891225444453084492e-01 5.90173779527698894042941901716404e-01 2.84708688341678414421287435231989e-03
            9.08147639531788897215847100596875e-01 6.24216082470002639470685323885846e-02 2.94307522212108388370843670145405e-02 3.04530716666442024739835936486543e-03
            9.08147639531850403571411334269214e-01 2.94307522211805228096181963337585e-02 6.24216082469690736189704693970270e-02 3.04530716666259922142945271161807e-03
            6.24216082469753116845900819953386e-02 9.08147639531814876434623329259921e-01 2.94307522212098326974683004664257e-02 3.04530716666247128557309942209486e-03
            6.24216082469878572047683462642453e-02 2.94307522211964094072111919331292e-02 9.08147639531815764613043029385153e-01 3.04530716666340239839882997330278e-03
            2.94307522211861953553846404929573e-02 9.08147639531878825280841738276649e-01 6.24216082469349897721144770912360e-02 3.04530716666041173512624595787202e-03
            2.94307522212155676932798797906798e-02 6.24216082469528851794926538332220e-02 9.08147639531831529779992706608027e-01 3.04530716666288848656907184420106e-03
            8.53441207153422753961535818234552e-01 1.17038555323248083528397955888067e-01 2.95202375233291625100662258773809e-02 4.11143575252731745955925291013955e-03
            8.53441207153414649333456054591807e-01 2.95202375233448027769256327701441e-02 1.17038555323240547889618312638049e-01 4.11143575253037230760044806743281e-03
            1.17038555323194598534186638971732e-01 8.53441207153465941637193736823974e-01 2.95202375233394320730440085753798e-02 4.11143575252700520933357708486255e-03
            1.17038555323250345607810629644518e-01 2.95202375233389845143872065591495e-02 8.53441207153410652530567404028261e-01 4.11143575252922045121239946752212e-03
            2.95202375233244475316585209156983e-02 8.53441207153490366543735490267863e-01 1.17038555323185189394052940770052e-01 4.11143575252849099999075122013892e-03
            2.95202375233212903349322431267865e-02 1.17038555323194598534186638971732e-01 8.53441207153484038272495126875583e-01 4.11143575252741200198869364612619e-03
            7.84561667827341380210270926909288e-01 1.84167433829063698080830135950237e-01 3.12708983435949217088989371404750e-02 5.02860503405233843360333523264671e-03
            7.84561667827341380210270926909288e-01 3.12708983435956086593954239560844e-02 1.84167433829063004191439745227399e-01 5.02860503405231674955988552255803e-03
            1.84167433828987842092672622129612e-01 7.84561667827407882569445973786060e-01 3.12708983436043030934570197132416e-02 5.02860503405243644547972792224755e-03
            1.84167433829062893169137282711745e-01 3.12708983435923959515179149093456e-02 7.84561667827344710879344802378910e-01 5.02860503405081274430621363080718e-03
            3.12708983436014859025320333785203e-02 7.84561667827366804317534842994064e-01 1.84167433829031668146569700184045e-01 5.02860503405542190458188400725703e-03
            3.12708983435994736232999002822908e-02 1.84167433829027227254471199557884e-01 7.84561667827373354633380131417653e-01 5.02860503405405060567412434124890e-03
            7.08519881821619734729722495103488e-01 2.60168754876507091200465993097168e-01 3.13113633018731740698115117993439e-02 5.62878756589628610751407933321389e-03
            7.08519881821622954376493908057455e-01 3.13113633018684209274873353479052e-02 2.60168754876508645512700468316325e-01 5.62878756589599380660837724121848e-03
            2.60168754876437591239124458297738e-01 7.08519881821694563761582230654312e-01 3.13113633018679005104445423057768e-02 5.62878756589763225293143733551915e-03
            2.60168754876498486972025148133980e-01 3.13113633018632098181655010193936e-02 7.08519881821638275454233735217713e-01 5.62878756589490960443589173678447e-03
            3.13113633018666237539662233757554e-02 7.08519881821631503093783521762816e-01 2.60168754876501817641099023603601e-01 5.62878756589649687641641051527586e-03
            3.13113633018663253815283553649351e-02 2.60168754876487051674871509021614e-01 7.08519881821646602126918423891766e-01 5.62878756589638325202873403441117e-03
            6.23662680919337741869412639061920e-01 3.44939109327516923997336562024429e-01 3.13982097531453341332507989136502e-02 6.27085339778674609456654565065037e-03
            6.23662680919329526219030412903521e-01 3.13982097531447859606323902426084e-02 3.44939109327525694759231100761099e-01 6.27085339778707308994176727878767e-03
            3.44939109327494275447634208831005e-01 6.23662680919366607668052893131971e-01 3.13982097531391168843128980370238e-02 6.27085339778907062402435457215688e-03
            3.44939109327520476711015362525359e-01 3.13982097531416357028000163609249e-02 6.23662680919337852891715101577574e-01 6.27085339778716849973294600317786e-03
            3.13982097531362372433427765372471e-02 6.23662680919338519025529876671499e-01 3.44939109327525250670021250698483e-01 6.27085339778543464361870718448699e-03
            3.13982097531376180832296540756943e-02 3.44939109327520476711015362525359e-01 6.23662680919341960716906214656774e-01 6.27085339778700456836446619490744e-03
            8.02460515514437200401687277917517e-01 1.25010019659880150522113240185718e-01 7.25294648256826490761994818967651e-02 6.53816728216865920841893711212833e-03
            8.02460515514371586220931931165978e-01 7.25294648257160112780894678508048e-02 1.25010019659912402500978600983217e-01 6.53816728216811537260921838310423e-03
            1.25010019659864191066134253560449e-01 8.02460515514425098970718863711227e-01 7.25294648257106544519956514704973e-02 6.53816728217038525827753403518727e-03
            1.25010019659891807863871804329392e-01 7.25294648257028690130354675602575e-02 8.02460515514405337000880535924807e-01 6.53816728216693402592207817747294e-03
            7.25294648256896296034668125685130e-02 8.02460515514371586220931931165978e-01 1.25010019659938742542237832822138e-01 6.53816728217031413461501898609640e-03
            7.25294648256799429075769580776978e-02 1.25010019659892085419627960618527e-01 8.02460515514427985550582889118232e-01 6.53816728216885956898041243334774e-03
            5.31386343848361364372578918846557e-01 4.36826791108310630296074350553681e-01 3.17868650433280053313467305997619e-02 6.70475295465790613325118840748473e-03
            5.31386343848350817253844979859423e-01 3.17868650433291086154774518490740e-02 4.36826791108320067191783664384275e-01 6.70475295465786276516428898730737e-03
            4.36826791108323175816252614822588e-01 5.31386343848351594409962217469001e-01 3.17868650433251742626339364505839e-02 6.70475295465877436235091479943549e-03
            4.36826791108322454171286608470837e-01 3.17868650433278804312564602696511e-02 5.31386343848349707030820354702882e-01 6.70475295465798159372239339859334e-03
            3.17868650433239946506702722217597e-02 5.31386343848359476993437056080438e-01 4.36826791108316569989256095141172e-01 6.70475295465666146915717504839449e-03
            3.17868650433250701792253778421582e-02 4.36826791108329004487131896894425e-01 5.31386343848345932272536629170645e-01 6.70475295465762597540981815313899e-03
            7.29210554614469153023037506500259e-01 1.95225741421010040577144195594883e-01 7.55637039645208063998182979048579e-02 8.10398375185674216258924218436732e-03
            7.29210554614469153023037506500259e-01 7.55637039645158242739952925148827e-02 1.95225741421015008825179393170401e-01 8.10398375185367690620719116623150e-03
            1.95225741420998188946356322048814e-01 7.29210554614469153023037506500259e-01 7.55637039645327135417574027087539e-02 8.10398375185792524399985836680571e-03
            1.95225741421000270614527494217327e-01 7.55637039645057212444712035903649e-02 7.29210554614494022018789110006765e-01 8.10398375185379833685050954272810e-03
            7.55637039645300906398617257764272e-02 7.29210554614403982931492009811336e-01 1.95225741421065968062009687855607e-01 8.10398375185522774899471443177390e-03
            7.55637039645257468922778798514628e-02 1.95225741421016812937594409049780e-01 7.29210554614457495681278942356585e-01 8.10398375185554173394386623385799e-03
            6.47395307940577979799456898035714e-01 2.77031342028366378560377825124306e-01 7.55733500310556416401652768399799e-02 9.21602869555110219834670459704284e-03
            6.47395307940592967810289337649010e-01 7.55733500310482725348393273634429e-02 2.77031342028358773532659142802004e-01 9.21602869555012034485930172422741e-03
            2.77031342028364047092026112295571e-01 6.47395307940594855189431200415129e-01 7.55733500310410422073914560314734e-02 9.21602869555086454123049577447091e-03
            2.77031342028360938467557161857258e-01 7.55733500310375727604395024172845e-02 6.47395307940601405505276488838717e-01 9.21602869555103974830156943198745e-03
            7.55733500310353661921780599186604e-02 6.47395307940563879967044158547651e-01 2.77031342028400739962989973719232e-01 9.21602869554641324079113928746665e-03
            7.55733500310362127372343366005225e-02 2.77031342028365767937714281288208e-01 6.47395307940597963813900150853442e-01 9.21602869554873083135504430174478e-03
            5.56614640862459930659156270849053e-01 3.67086174365095052607443903980311e-01 7.62991847724450167333998251706362e-02 1.00221108897942754595300485220832e-02
            5.56614640862468257331840959523106e-01 7.62991847724459326673951409247820e-02 3.67086174365085837756339515181025e-01 1.00221108897945963833731042313957e-02
            3.67086174365103157235523667623056e-01 5.56614640862462817239020296256058e-01 7.62991847724340255254560361208860e-02 1.00221108897939874954330363721056e-02
            3.67086174365103212746674898880883e-01 7.62991847724408395192696730191528e-02 5.56614640862455933856267620285507e-01 1.00221108897952694560817832325483e-02
            7.62991847724286409437866041116649e-02 5.56614640862459930659156270849053e-01 3.67086174365111483908208356297109e-01 1.00221108897912761226400846226170e-02
            7.62991847724316246681652842198673e-02 3.67086174365097550609249310582527e-01 5.56614640862470810844797597383149e-01 1.00221108897935624881814220543674e-02
            6.55804993103131339537981148168910e-01 2.07677817553296312214072827373457e-01 1.36517189343572348247946024457633e-01 1.08925884826709586800097540049137e-02
            6.55804993103188182956841956183780e-01 1.36517189343557499014991662988905e-01 2.07677817553254318028166380827315e-01 1.08925884826699022334128841293932e-02
            2.07677817553273053041706930343935e-01 6.55804993103150879463214550924022e-01 1.36517189343576039739502903103130e-01 1.08925884826687122131083640397264e-02
            2.07677817553268279082701042170811e-01 1.36517189343538458690119341554237e-01 6.55804993103193289982755231903866e-01 1.08925884826687243561726958773761e-02
            1.36517189343568823289842839585617e-01 6.55804993103131450560283610684564e-01 2.07677817553299726149873549729818e-01 1.08925884826678361777529957521438e-02
            1.36517189343564021575261335783580e-01 2.07677817553268306838276657799724e-01 6.55804993103167643830886390787782e-01 1.08925884826666912602588510594614e-02
            5.70923789561694650096512759773759e-01 2.91806857017915921570505588533706e-01 1.37269353420389428332981651692535e-01 1.22687734810442063831947834273706e-02
            5.70923789561736949593750978237949e-01 1.37269353420383682928829216507438e-01 2.91806857017879339721844189625699e-01 1.22687734810456566120207000381015e-02
            2.91806857017880116877961427235277e-01 5.70923789561755157251354830805212e-01 1.37269353420364725870683741959510e-01 1.22687734810411792907292038989908e-02
            2.91806857017890386440939209933276e-01 1.37269353420368556140118698749575e-01 5.70923789561740946396639628801495e-01 1.22687734810440936261688449349094e-02
            1.37269353420353928951769262312155e-01 5.70923789561736949593750978237949e-01 2.91806857017909093698904143820982e-01 1.22687734810410040836581302414743e-02
            1.37269353420356593487028362687852e-01 2.91806857017874565762838301452575e-01 5.70923789561768812994557720230659e-01 1.22687734810412989866490462986803e-02
            4.78028270406235411549289437971311e-01 3.83097780854575353792768055427587e-01 1.38873948739189234657942506601103e-01 1.31829715331267232486522900103409e-02
            4.78028270406262334457636598017416e-01 1.38873948739193980861372779145313e-01 3.83097780854543601414263775950531e-01 1.31829715331277866341430637930898e-02
            3.83097780854526059890474698477192e-01 4.78028270406294142347292108752299e-01 1.38873948739179797762233192770509e-01 1.31829715331257379257179351839113e-02
            3.83097780854547487194849963998422e-01 1.38873948739189817525030434808286e-01 4.78028270406262723035695216822205e-01 1.31829715331272107059490394931345e-02
            1.38873948739170527399977572713397e-01 4.78028270406264887970593235877459e-01 3.83097780854564529118277960151318e-01 1.31829715331264057942561862546427e-02
            1.38873948739172498045846282366256e-01 3.83097780854534442074310618409072e-01 4.78028270406293087635418714853586e-01 1.31829715331268169237199927579240e-02
            4.88026817278799873722761049066321e-01 2.97907118579997409746340508718276e-01 2.14066064141202772042049673473230e-01 1.51516998095694455789272936385714e-02
            4.88026817278788882514817260016571e-01 2.14066064141203715731620604856289e-01 2.97907118580007346242410903869313e-01 1.51516998095690431230808670193255e-02
            2.97907118579970653371447042445652e-01 4.88026817278850555403835187462391e-01 2.14066064141178791224717770091956e-01 1.51516998095675391178271951275747e-02
            2.97907118579965990434743616788182e-01 2.14066064141187561986612308828626e-01 4.88026817278846447578644074383192e-01 1.51516998095677056512808889010557e-02
            2.14066064141161527256684848907753e-01 4.88026817278841729130789417467895e-01 2.97907118579996688101374502366525e-01 1.51516998095681428015968350564435e-02
            2.14066064141163109324494939755823e-01 2.97907118579965990434743616788182e-01 4.88026817278870872485185827827081e-01 1.51516998095689112840966927819863e-02
            ];
        
    case  28
        
        % ALG. DEG.:   28
        % PTS CARD.:  171
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.3e-16
        
        xyw_bar=[
            3.52781468614848242593140525968920e-03 3.52781468614803963776416217967835e-03 9.92944370627703465359559231728781e-01 1.68828471789717960219179282432833e-04
            3.52781468614818492085527523727251e-03 9.92944370627703354337256769213127e-01 3.52781468614848936482530916691758e-03 1.68828471789721592296457108872687e-04
            9.92944370627703354337256769213127e-01 3.52781468614843515471668489169588e-03 3.52781468614821050802654589517715e-03 1.68828471789724817797920253248378e-04
            2.32544785996821011531121570214964e-02 2.32544785996821011531121570214964e-02 9.53491042800635790754881782049779e-01 1.29903671202049328728400645616148e-03
            2.32544785996818652307194241757315e-02 9.53491042800635790754881782049779e-01 2.32544785996823266671640340064187e-02 1.29903671202048851679444751994197e-03
            9.53491042800635790754881782049779e-01 2.32544785996819727835749347377714e-02 2.32544785996822364615432832124498e-02 1.29903671202050174406095184309606e-03
            6.13061711490497721066539327239298e-02 6.13061711490494321008526412697393e-02 8.77387657701900747220236098655732e-01 3.93842260805571820619919165551437e-03
            6.13061711490498553733807796106703e-02 8.77387657701900081086421323561808e-01 6.13061711490500149679405694769230e-02 3.93842260805573989024264136560305e-03
            8.77387657701900636197933636140078e-01 6.13061711490494321008526412697393e-02 6.13061711490499317012137225901824e-02 3.93842260805572341036961958593565e-03
            4.85597365316512985522479084465886e-01 4.85597365316513485122840165786329e-01 2.88052693669735293546807497477857e-02 5.79050543361936832092995430798510e-03
            4.85597365316512985522479084465886e-01 2.88052693669737236437100591501803e-02 4.85597365316513318589386472012848e-01 5.79050543361942209735770958900503e-03
            2.88052693669737791548612904080073e-02 4.85597365316512985522479084465886e-01 4.85597365316513207567084009497194e-01 5.79050543361941168901685372816246e-03
            1.19574379687203399824646510296589e-01 1.19574379687203594113675819698983e-01 7.60851240625592950550526438746601e-01 7.52896849588353782312122319808623e-03
            1.19574379687202816957558582089405e-01 7.60851240625593949751248601387488e-01 1.19574379687203191657829393079737e-01 7.52896849588350226128996567354079e-03
            7.60851240625593283617433826293563e-01 1.19574379687203302680131855595391e-01 1.19574379687203413702434318111045e-01 7.52896849588348578141694389387339e-03
            1.93832849735945389468838584434707e-01 1.93832849735944889868477503114264e-01 6.12334300528109665151532681193203e-01 1.09520031775944982205261979402167e-02
            1.93832849735944195979087112391426e-01 6.12334300528110331285347456287127e-01 1.93832849735945500491141046950361e-01 1.09520031775945589358478571284650e-02
            6.12334300528110886396859768865397e-01 1.93832849735944889868477503114264e-01 1.93832849735944223734662728020339e-01 1.09520031775945034246966258706379e-02
            4.36854825801193724643667337659281e-01 4.36854825801192891976398868791875e-01 1.26290348397613383379933793548844e-01 1.13144540852838822175829847083151e-02
            4.36854825801192891976398868791875e-01 1.26290348397614132780475415529509e-01 4.36854825801193058509852562565356e-01 1.13144540852839117078820763140357e-02
            1.26290348397614382580655956189730e-01 4.36854825801192558909491481244913e-01 4.36854825801193114021003793823184e-01 1.13144540852838527272838931025944e-02
            2.74786691854845666060924713747227e-01 2.74786691854845777083227176262881e-01 4.50426616290308612366999341247720e-01 1.42362905099713173140418831508214e-02
            2.74786691854845999127832101294189e-01 4.50426616290308667878150572505547e-01 2.74786691854845277482866094942437e-01 1.42362905099712271084211323568525e-02
            4.50426616290308556855848109989893e-01 2.74786691854846387705890720098978e-01 2.74786691854845055438261169911129e-01 1.42362905099712722112315077538369e-02
            3.61886974590005627838706914189970e-01 3.61886974590005849883311839221278e-01 2.76226050819988522277981246588752e-01 1.49105703269718017911493745941698e-02
            3.61886974590005627838706914189970e-01 2.76226050819989354945249715456157e-01 3.61886974590005017216043370353873e-01 1.49105703269718625064710337824181e-02
            2.76226050819989632501005871745292e-01 3.61886974590005461305253220416489e-01 3.61886974590004906193740907838219e-01 1.49105703269718087300432785013982e-02
            9.76405945543714515189037683740025e-01 1.90656116400245469189034253076898e-02 4.52844281626093789205889095228486e-03 4.89575022817241383397968146340418e-04
            9.76405945543714515189037683740025e-01 4.52844281626076268498781729476832e-03 1.90656116400247238606979749420134e-02 4.89575022817250165435565278926333e-04
            1.90656116400246301856302721944303e-02 9.76405945543714737233642608771333e-01 4.52844281626063605017407098785043e-03 4.89575022817241058137316400689087e-04
            1.90656116400246301856302721944303e-02 4.52844281626113478317341431989007e-03 9.76405945543714293144432758708717e-01 4.89575022817266428468152561492843e-04
            4.52844281626104717963787749113180e-03 9.76405945543714515189037683740025e-01 1.90656116400244046715783952095080e-02 4.89575022817249623334479036174116e-04
            4.52844281626053630357420232144250e-03 1.90656116400246301856302721944303e-02 9.76405945543714848255945071286988e-01 4.89575022817247780190785810816578e-04
            9.49626039031084334673948887939332e-01 4.57527920669594106994892968032218e-02 4.62116890195625462656181525744614e-03 7.70464298658109637656099533131737e-04
            9.49626039031083335473226725298446e-01 4.62116890195622860570967560533973e-03 4.57527920669604376557870750730217e-02 7.70464298658118311273479417167209e-04
            4.57527920669597021330332609068137e-02 9.49626039031083668540134112845408e-01 4.62116890195662932683262624777853e-03 7.70464298658138369013670398999238e-04
            4.57527920669602294889699578561704e-02 4.62116890195593717216571150174786e-03 9.49626039031083779562436575361062e-01 7.70464298658085026266784112181085e-04
            4.62116890195595972357089920024009e-03 9.49626039031084334673948887939332e-01 4.57527920669597021330332609068137e-02 7.70464298658087628351998077391727e-04
            4.62116890195662672474741228256789e-03 4.57527920669607637838005587127554e-02 9.49626039031082558317109487688867e-01 7.70464298658162329881682328647230e-04
            9.09575444421073742162775488395710e-01 8.55585301859531771606626193715783e-02 4.86602539297308067656189223271213e-03 1.19205191231310284570577007201564e-03
            9.09575444421073298073565638333093e-01 4.86602539297293842923686213453038e-03 8.55585301859537600277505475787621e-02 1.19205191231306771755538154167198e-03
            8.55585301859525665379990755354811e-02 9.09575444421074630341195188520942e-01 4.86602539297276148744231250020675e-03 1.19205191231306403126799509095690e-03
            8.55585301859539404389920491666999e-02 4.86602539297307807447667826750148e-03 9.09575444421072965006658250786131e-01 1.19205191231309547313099717058549e-03
            4.86602539297305639043322855741280e-03 9.09575444421073298073565638333093e-01 8.55585301859536073720846616197377e-02 1.19205191231310284570577007201564e-03
            4.86602539297277363050664433785641e-03 8.55585301859539404389920491666999e-02 9.09575444421073298073565638333093e-01 1.19205191231303909461802792435492e-03
            8.53928600117039415806630131555721e-01 1.40745793360055215792669969232520e-01 5.32560652290536840069989921175875e-03 1.69230236401778284284491693512109e-03
            8.53928600117039526828932594071375e-01 5.32560652290524263324789089324440e-03 1.40745793360055243548245584861434e-01 1.69230236401774641365192142217211e-03
            1.40745793360055188037094353603607e-01 8.53928600117039526828932594071375e-01 5.32560652290525737839743669610471e-03 1.69230236401777655447231651919537e-03
            1.40745793360055049259216275459039e-01 5.32560652290533457359211766402041e-03 8.53928600117039526828932594071375e-01 1.69230236401776224300363971053685e-03
            5.32560652290529814439912215107142e-03 8.53928600117039415806630131555721e-01 1.40745793360055326814972431748174e-01 1.69230236401776224300363971053685e-03
            5.32560652290531115482519197712463e-03 1.40745793360055854170909128697531e-01 8.53928600117038860695117818977451e-01 1.69230236401777091662101959457232e-03
            7.86232363633783681855504710256355e-01 2.08266834439758047325952361461532e-01 5.50080192645827081854292828211328e-03 2.03759884643626640321478760142782e-03
            7.86232363633783681855504710256355e-01 5.50080192645838010612191482096023e-03 2.08266834439757936303649898945878e-01 2.03759884643627854627911943907748e-03
            2.08266834439758491415162211524148e-01 7.86232363633783126743992397678085e-01 5.50080192645840959642100642668083e-03 2.03759884643629979664170015496438e-03
            2.08266834439757991814801130203705e-01 5.50080192645833153386458747036158e-03 7.86232363633783681855504710256355e-01 2.03759884643628114836433340428812e-03
            5.50080192645825086922295454883169e-03 7.86232363633783681855504710256355e-01 2.08266834439758019570376745832618e-01 2.03759884643625599487393174058525e-03
            5.50080192645841219850622039189147e-03 2.08266834439757991814801130203705e-01 7.86232363633783570833202247740701e-01 2.03759884643627897995998843327925e-03
            7.10218209945296274909765088523272e-01 2.84299186848130214233520973721170e-01 5.48260320657351085671393775555771e-03 2.22726029161563528896428110215311e-03
            7.10218209945296607976672476070235e-01 5.48260320657365136931549187693236e-03 2.84299186848129714633159892400727e-01 2.22726029161566824871032466148790e-03
            2.84299186848130214233520973721170e-01 7.10218209945296163887462626007618e-01 5.48260320657367739016763152903877e-03 2.22726029161565046779469589921518e-03
            2.84299186848130491789277130010305e-01 5.48260320657360106233468854952662e-03 7.10218209945295941842857700976310e-01 2.22726029161565133515643388761873e-03
            5.48260320657359759288773659591243e-03 7.10218209945295941842857700976310e-01 2.84299186848130491789277130010305e-01 2.22726029161564309521992299778503e-03
            5.48260320657359152135557067708760e-03 2.84299186848129603610857429885073e-01 7.10218209945296830021277401101543e-01 2.22726029161563615632601909055666e-03
            9.19615743255533413247349017183296e-01 5.56921540915342391797615562154533e-02 2.46921026529323475728894266012503e-02 2.22811737200658173396750072470240e-03
            9.19615743255533413247349017183296e-01 2.46921026529322192033522043175253e-02 5.56921540915343710187457304527925e-02 2.22811737200656308569013397402614e-03
            5.56921540915337326405065709877817e-02 9.19615743255533413247349017183296e-01 2.46921026529328679899322196433786e-02 2.22811737200657262666925184646516e-03
            5.56921540915340518296261507202871e-02 2.46921026529317196029911229970821e-02 9.19615743255534190403466254792875e-01 2.22811737200655137630667113057825e-03
            2.46921026529319034836795765386341e-02 9.19615743255533413247349017183296e-01 5.56921540915347179634409258142114e-02 2.22811737200658346869097670150950e-03
            2.46921026529327673759706129885672e-02 5.56921540915341767297164210503979e-02 9.19615743255533080180441629636334e-01 2.22811737200656221832839598562259e-03
            6.28121500758708983802591774292523e-01 3.66399643439952127366154854826164e-01 5.47885580133888883125337088131346e-03 2.37598555928124951369673212298039e-03
            6.28121500758709094824894236808177e-01 5.47885580133890270904117869577021e-03 3.66399643439952016343852392310509e-01 2.37598555928129244810276254895598e-03
            3.66399643439951738788096236021374e-01 6.28121500758709427891801624355139e-01 5.47885580133888883125337088131346e-03 2.37598555928126122308019496642828e-03
            3.66399643439952127366154854826164e-01 5.47885580133884372844299548432900e-03 6.28121500758708983802591774292523e-01 2.37598555928124344216456620415556e-03
            5.47885580133884546316647146113610e-03 6.28121500758709094824894236808177e-01 3.66399643439952016343852392310509e-01 2.37598555928125211578194608819103e-03
            5.47885580133881163605868991339776e-03 3.66399643439952127366154854826164e-01 6.28121500758709094824894236808177e-01 2.37598555928128767761320361273647e-03
            5.41474950946130073603512755653355e-01 4.53014671446369310281454545474844e-01 5.51037760750061611503269887180068e-03 2.48742263085009870579589197348014e-03
            5.41474950946129407469697980559431e-01 5.51037760750079045474203454091366e-03 4.53014671446369809881815626795287e-01 2.48742263085014380860626737046459e-03
            4.53014671446368977214547157927882e-01 5.41474950946130295648117680684663e-01 5.51037760750072713733516138745472e-03 2.48742263085011214990283079373512e-03
            4.53014671446368533125337307865266e-01 5.51037760750075576027251500477178e-03 5.41474950946130739737327530747280e-01 2.48742263085011084886022381112980e-03
            5.51037760750074274984644517871857e-03 5.41474950946130850759629993262934e-01 4.53014671446368422103034845349612e-01 2.48742263085011084886022381112980e-03
            5.51037760750075489291077701636823e-03 4.53014671446370031926420551826595e-01 5.41474950946129185425093055528123e-01 2.48742263085014077284018441105218e-03
            8.70847043597263614778114515502239e-01 1.02747376222606420781424674260052e-01 2.64055801801299644404608102377097e-02 3.24701241373825683772524897108269e-03
            8.70847043597263614778114515502239e-01 2.64055801801296730069168461341178e-02 1.02747376222606712214968638363644e-01 3.24701241373823298527745428998514e-03
            1.02747376222606059958941671084176e-01 8.70847043597264613978836678143125e-01 2.64055801801292844288582273293287e-02 3.24701241373825119987395204645964e-03
            1.02747376222606198736819749228744e-01 2.64055801801302836295803899702150e-02 8.70847043597263503755812052986585e-01 3.24701241373825423564003500587205e-03
            2.64055801801301101572327922895056e-02 8.70847043597263170688904665439622e-01 1.02747376222606767726119869621471e-01 3.24701241373821997485138446393194e-03
            2.64055801801291629982149089528320e-02 1.02747376222606198736819749228744e-01 8.70847043597264613978836678143125e-01 3.24701241373822604638355038275677e-03
            8.08055925125646168361015497794142e-01 1.63625151528200618589181658535381e-01 2.83189233461532130498028436704772e-02 4.21183327129194279986323934394932e-03
            8.08055925125645946316410572762834e-01 2.83189233461530534552430538042245e-02 1.63625151528201007167240277340170e-01 4.21183327129192371790500359907128e-03
            1.63625151528201479012025743031700e-01 8.08055925125645280182595797668910e-01 2.83189233461532685609540749283042e-02 4.21183327129202086241965830026857e-03
            1.63625151528200368789001117875159e-01 2.83189233461531228441820928765082e-02 8.08055925125646501427922885341104e-01 4.21183327129195407556583319319543e-03
            2.83189233461529042690241197988144e-02 8.08055925125646723472527810372412e-01 1.63625151528200341033425502246246e-01 4.21183327129187688037115222527973e-03
            2.83189233461536050973084144288805e-02 1.63625151528201173700693971113651e-01 8.08055925125645169160293335153256e-01 4.21183327129202086241965830026857e-03
            7.37116338815208438361992193676997e-01 2.34175394401963360424545612659131e-01 2.87082667828282012134621936638723e-02 4.78367400458557748971211154298544e-03
            7.37116338815208438361992193676997e-01 2.87082667828285863220738605150473e-02 2.34175394401962971846486993854342e-01 4.78367400458557488762689757777480e-03
            2.34175394401964304114116544042190e-01 7.37116338815206995072060180973494e-01 2.87082667828286730582476593554020e-02 4.78367400458560351056425119509186e-03
            2.34175394401963415935696843916958e-01 2.87082667828283261135524639939831e-02 7.37116338815208216317387268645689e-01 4.78367400458561478626684504433797e-03
            2.87082667828280589661371635656906e-02 7.37116338815208993473504506255267e-01 2.34175394401962999602062609483255e-01 4.78367400458558095915906349659963e-03
            2.87082667828285308109226292572203e-02 2.34175394401963304913394381401304e-01 7.37116338815208105295084806130035e-01 4.78367400458554626468954396045774e-03
            6.59255694951647930324156732240226e-01 3.12168255169341768429092098813271e-01 2.85760498790103012467511689465027e-02 5.22135104863136190189720764465164e-03
            6.59255694951648818502576432365458e-01 2.85760498790107037025975955657486e-02 3.12168255169340491672613779883250e-01 5.22135104863138879011108528516161e-03
            3.12168255169341601895638405039790e-01 6.59255694951647819301854269724572e-01 2.85760498790105232913560939778108e-02 5.22135104863134715674766184179134e-03
            3.12168255169341601895638405039790e-01 2.85760498790105232913560939778108e-02 6.59255694951647819301854269724572e-01 5.22135104863135929981199367944100e-03
            2.85760498790105232913560939778108e-02 6.59255694951647930324156732240226e-01 3.12168255169341546384487173781963e-01 5.22135104863141047415453499525029e-03
            2.85760498790103151245389767609595e-02 3.12168255169340769228369936172385e-01 6.59255694951648929524878894881113e-01 5.22135104863134195257723391137006e-03
            8.21610770110138166621993605076568e-01 1.13243304903654681825564409791696e-01 6.51459249862071515524419851317361e-02 5.46922489919737450647652110546915e-03
            8.21610770110138166621993605076568e-01 6.51459249862066797076565194402065e-02 1.13243304903655153670349875483225e-01 5.46922489919734848562438145336273e-03
            1.13243304903655458981681647401274e-01 8.21610770110138166621993605076568e-01 6.51459249862064160296881709655281e-02 5.46922489919740920094604064161103e-03
            1.13243304903654931625744950451917e-01 6.51459249862074013526225257919577e-02 8.21610770110137611510481292498298e-01 5.46922489919735368979480938378401e-03
            6.51459249862070127745639069871686e-02 8.21610770110138166621993605076568e-01 1.13243304903654862236805911379633e-01 5.46922489919728169877055634628960e-03
            6.51459249862063743963247475221578e-02 1.13243304903654806725654680121806e-01 8.21610770110138721733505917654838e-01 5.46922489919742481345732443287488e-03
            5.74541105959161191307771332503762e-01 3.96768611430950823848462505338830e-01 2.86902826098879848437661621574080e-02 5.62170007491155383078096363647091e-03
            5.74541105959161968463888570113340e-01 2.86902826098883213801205016579843e-02 3.96768611430949713625437880182290e-01 5.62170007491166398572168816372141e-03
            3.96768611430950157714647730244906e-01 5.74541105959161635396981182566378e-01 2.86902826098882623995223184465431e-02 5.62170007491158245371831725378797e-03
            3.96768611430950157714647730244906e-01 2.86902826098882103578180391423302e-02 5.74541105959161635396981182566378e-01 5.62170007491157724954788932336669e-03
            2.86902826098882762773101262609998e-02 5.74541105959161080285468869988108e-01 3.96768611430950657315008811565349e-01 5.62170007491160500512350495228020e-03
            2.86902826098882068883710871887160e-02 3.96768611430949935670042805213598e-01 5.74541105959161857441586107597686e-01 5.62170007491162755652869265077243e-03
            7.52736321001383545947760467242915e-01 1.78349437979342606563903927963111e-01 6.89142410192738474883356047939742e-02 6.82219197720224622416562709759091e-03
            7.52736321001383879014667854789877e-01 6.89142410192738613661234126084310e-02 1.78349437979342273496996540416148e-01 6.82219197720216469216225618765748e-03
            1.78349437979343217186567471799208e-01 7.52736321001382435724735842086375e-01 6.89142410192743470886966861144174e-02 6.82219197720225836722995893524057e-03
            1.78349437979342689830630774849851e-01 6.89142410192736670770941032060364e-02 7.52736321001383656970062929758569e-01 6.82219197720224709152736508599446e-03
            6.89142410192731813545208297000499e-02 7.52736321001384656170785092399456e-01 1.78349437979342217985845309158321e-01 6.82219197720217683522658802530714e-03
            6.89142410192748328112699596204038e-02 1.78349437979342745341782006107678e-01 7.52736321001382435724735842086375e-01 6.82219197720226790820907680767959e-03
            6.77869789721086690725826429115841e-01 2.52933135839176648218540321977343e-01 6.91970744397366610556332489068154e-02 7.63578348187659173423869063412894e-03
            6.77869789721087134815036279178457e-01 6.91970744397375631118407568465045e-02 2.52933135839175315950910771789495e-01 7.63578348187657872381262080807574e-03
            2.52933135839176370662784165688208e-01 6.77869789721086135614314116537571e-01 6.91970744397374382117504865163937e-02 7.63578348187656658074828897042607e-03
            2.52933135839176481685086628203862e-01 6.91970744397370357559040598971478e-02 6.77869789721086579703523966600187e-01 7.63578348187658739743000069211121e-03
            6.91970744397366888112088645357289e-02 6.77869789721087134815036279178457e-01 2.52933135839176204129330471914727e-01 7.63578348187662122453778223984955e-03
            6.91970744397369663669650208248640e-02 2.52933135839176037595876778141246e-01 6.77869789721087023792733816662803e-01 7.63578348187653362100224541109128e-03
            5.96181882441887966983529167919187e-01 3.34657518177715052054566058359342e-01 6.91605993803969809619047737214714e-02 8.25598265452889140680881041589600e-03
            5.96181882441889077206553793075727e-01 6.91605993803976332179317410009389e-02 3.34657518177713275697726658108877e-01 8.25598265452900936800517683877843e-03
            3.34657518177714441431902514523244e-01 5.96181882441888189028134092950495e-01 6.91605993803973140288121612684336e-02 8.25598265452895906102437351137269e-03
            3.34657518177714385920751283265417e-01 6.91605993803973279065999690828903e-02 5.96181882441888300050436555466149e-01 8.25598265452891048876704616077404e-03
            6.91605993803974805622658550419146e-02 5.96181882441887966983529167919187e-01 3.34657518177714607965356208296726e-01 8.25598265452894518323656569691593e-03
            6.91605993803971058619950440515822e-02 3.34657518177714385920751283265417e-01 5.96181882441888522095041480497457e-01 8.25598265452900069438779695474295e-03
            5.09343076403983130440167315100553e-01 4.21087880063123531915181274598581e-01 6.95690435328933376446514103008667e-02 8.62969827541772670220776575433774e-03
            5.09343076403983130440167315100553e-01 6.95690435328939205117393385080504e-02 4.21087880063122921292517730762484e-01 8.62969827541790364400231538866137e-03
            4.21087880063123587426332505856408e-01 5.09343076403982908395562390069244e-01 6.95690435328935041781051040743478e-02 8.62969827541778915225290091939314e-03
            4.21087880063123032314820193278138e-01 6.95690435328938372450124916213099e-02 5.09343076403983130440167315100553e-01 8.62969827541782905089284838595631e-03
            6.95690435328939899006783775803342e-02 5.09343076403983130440167315100553e-01 4.21087880063122921292517730762484e-01 8.62969827541778394808247298897186e-03
            6.95690435328939343895271463225072e-02 4.21087880063123753959786199629889e-01 5.09343076403982353284050077490974e-01 8.62969827541784986757456010764145e-03
            6.87533646543294296726855918677757e-01 1.87462735345928138697146891900047e-01 1.25003618110777564575997189422196e-01 9.19068281131948071338566563781569e-03
            6.87533646543294296726855918677757e-01 1.25003618110778008665207039484812e-01 1.87462735345927694607937041837431e-01 9.19068281131947897866218966100860e-03
            1.87462735345927000718546651114593e-01 6.87533646543294296726855918677757e-01 1.25003618110778647043446198949823e-01 9.19068281131949632589694942907954e-03
            1.87462735345928527275205510704836e-01 1.25003618110777786620602114453504e-01 6.87533646543293741615343606099486e-01 9.19068281131954836760122873329237e-03
            1.25003618110777009464484876843926e-01 6.87533646543294296726855918677757e-01 1.87462735345928721564234820107231e-01 9.19068281131959520513508010708392e-03
            1.25003618110778952354777970867872e-01 1.87462735345928055430420045013307e-01 6.87533646543292964459226368489908e-01 9.19068281131944775363962207848090e-03
            6.09882363677961891745837874623248e-01 2.64906647850886134420989037607796e-01 1.25210988471151973833173087768955e-01 1.02528112373969550880081769150820e-02
            6.09882363677961114589720637013670e-01 1.25210988471153278345227022327890e-01 2.64906647850885579309476725029526e-01 1.02528112373970366200115478250154e-02
            2.64906647850885079709115643709083e-01 6.09882363677961891745837874623248e-01 1.25210988471153084056197712925496e-01 1.02528112373970609061402115003148e-02
            2.64906647850886411976745193896932e-01 1.25210988471152556700261015976139e-01 6.09882363677961114589720637013670e-01 1.02528112373969637616255567991175e-02
            1.25210988471152390166807322202658e-01 6.09882363677960448455905861919746e-01 2.64906647850887133621711200248683e-01 1.02528112373970400894584997786296e-02
            1.25210988471152140366626781542436e-01 2.64906647850885690331779187545180e-01 6.09882363677962224812745262170210e-01 1.02528112373969741699664126599600e-02
            5.25254911987723116517656762880506e-01 3.49006616007701941928331734743551e-01 1.25738472004574941554011502375943e-01 1.10260746354600737417772648996106e-02
            5.25254911987722339361539525270928e-01 1.25738472004575940754733665016829e-01 3.49006616007701719883726809712243e-01 1.10260746354601743557388715544221e-02
            3.49006616007701941928331734743551e-01 5.25254911987722672428446912817890e-01 1.25738472004575441154372583696386e-01 1.10260746354602107849318670673711e-02
            3.49006616007701941928331734743551e-01 1.25738472004575635443401893098780e-01 5.25254911987722450383841987786582e-01 1.10260746354601014973528805285241e-02
            1.25738472004575940754733665016829e-01 5.25254911987721340160817362630041e-01 3.49006616007702663573297741095303e-01 1.10260746354600650681598850155751e-02
            1.25738472004575579932250661840953e-01 3.49006616007701775394878040970070e-01 5.25254911987722672428446912817890e-01 1.10260746354601726210153955776150e-02
            5.33207332734685901520776951656444e-01 2.72179046638476807018491854250897e-01 1.94613620626837291460731194092659e-01 1.23917534199638693787060006457068e-02
            5.33207332734684902320054789015558e-01 1.94613620626838929039692516198556e-01 2.72179046638476140884677079156972e-01 1.23917534199639092773459481122700e-02
            2.72179046638476918040794316766551e-01 5.33207332734684791297752326499904e-01 1.94613620626838290661453356733546e-01 1.23917534199639422370919916716048e-02
            2.72179046638476918040794316766551e-01 1.94613620626838179639150894217892e-01 5.33207332734684902320054789015558e-01 1.23917534199638572356416688080571e-02
            1.94613620626838179639150894217892e-01 5.33207332734684902320054789015558e-01 2.72179046638476918040794316766551e-01 1.23917534199638242758956252487224e-02
            1.94613620626837791061092275413102e-01 2.72179046638477140085399241797859e-01 5.33207332734685013342357251531212e-01 1.23917534199639751968380352309396e-02
            4.46455240965652422335807614217629e-01 3.57596143740140481881439882272389e-01 1.95948615294207151293903734767810e-01 1.33365742525531766338175998498627e-02
            4.46455240965651589668539145350223e-01 1.95948615294208261516928359924350e-01 3.57596143740140148814532494725427e-01 1.33365742525532338796923070844969e-02
            3.57596143740141314548708351139794e-01 4.46455240965651312112782989061088e-01 1.95948615294207317827357428541291e-01 1.33365742525532061241166914555834e-02
            3.57596143740140592903742344788043e-01 1.95948615294208011716747819264128e-01 4.46455240965651367623934220318915e-01 1.33365742525532234713514512236543e-02
            1.95948615294208455805957669326745e-01 4.46455240965651645179690376608050e-01 3.57596143740139871258776338436292e-01 1.33365742525531263268367965224570e-02
            1.95948615294208261516928359924350e-01 3.57596143740141036992952194850659e-01 4.46455240965650701490119445224991e-01 1.33365742525532217366279752468472e-02
            ];
        
    case  30
        
        % ALG. DEG.:   30
        % PTS CARD.:  190
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.4e-16
        
        xyw_bar=[
            3.33333333333334924653001962724375e-01 3.33333333333335590786816737818299e-01 3.33333333333329484560181299457327e-01 1.37773483475028268863127323129447e-02
            2.60849100043316489694333881743660e-03 2.60849100043398151801965489937629e-03 9.94783017999132823661057045683265e-01 1.04046024904101539918996333078383e-04
            2.60849100042975139482348545527657e-03 9.94783017999138041709272783919005e-01 2.60849100043225590184192697051913e-03 1.04046024903988173029335817521002e-04
            9.94783017999136043307828458637232e-01 2.60849100043205120447176170728198e-03 2.60849100043190548769977965548605e-03 1.04046024904035783057235087234460e-04
            1.99081215329376323697196937700937e-02 1.99081215329428608262762878666763e-02 9.60183756934119503334557066409616e-01 9.38574375977459414432946260831159e-04
            1.99081215329434957350684953780728e-02 9.60183756934114285286341328173876e-01 1.99081215329421912230145608191378e-02 9.38574375977355005763735906754164e-04
            9.60183756934114285286341328173876e-01 1.99081215329420559145834346281845e-02 1.99081215329436587990752371979397e-02 9.38574375977383303440437778419891e-04
            4.97508711341201725897320784497424e-01 4.97508711341202891631496640911791e-01 4.98257731759532696003134333295748e-03 2.14981439621547872065310258449244e-03
            4.97508711341204723499487272420083e-01 4.98257731759566523110915881034089e-03 4.97508711341199671984725227957824e-01 2.14981439621544055673663109473637e-03
            4.98257731759559237272316778444292e-03 4.97508711341201725897320784497424e-01 4.97508711341202725098042947138310e-01 2.14981439621537030043585403404904e-03
            5.38543920770780878970462879351544e-02 5.38543920770771372685814526448667e-02 8.92291215845844809528841778956121e-01 3.20344050830286128025981007283463e-03
            5.38543920770712045142936119646038e-02 8.92291215845843921350422078830888e-01 5.38543920770848671963904052972794e-02 3.20344050830271686453043500364402e-03
            8.92291215845845031573446703987429e-01 5.38543920770740355830064061137818e-02 5.38543920770809328435468898987892e-02 3.20344050830283092259898047871047e-03
            1.10234628175727419896468006754731e-01 1.10234628175722632059674310767150e-01 7.79530743648550017432796721550403e-01 6.25977580797623840702659236967520e-03
            1.10234628175712945363784456276335e-01 7.79530743648566559755863636382855e-01 1.10234628175720494880351907340810e-01 6.25977580797610049651025221351119e-03
            7.79530743648562229886067598272348e-01 1.10234628175719370779539474369813e-01 1.10234628175718399334392927357840e-01 6.25977580797602590340078521080613e-03
            4.68466155978488285605010332801612e-01 4.68466155978493226097469914748217e-01 6.30676880430184882975197524501709e-02 7.53429356032105004142840343206444e-03
            4.68466155978494502853948233678238e-01 6.30676880430202230209957292572653e-02 4.68466155978485288002843844878953e-01 7.53429356032072131132970582712005e-03
            6.30676880430198483207249182669329e-02 4.68466155978490950140269433177309e-01 4.68466155978489173783430032926844e-01 7.53429356032101014278845596550127e-03
            1.73692667715927590021962600985717e-01 1.73692667715930587624129088908376e-01 6.52614664568141766842757078848081e-01 9.13427474706155449746258767618201e-03
            1.73692667715933224403812573655159e-01 6.52614664568134550393097015330568e-01 1.73692667715932280714241642272100e-01 9.13427474706150939465221227919756e-03
            6.52614664568138658218288128409768e-01 1.73692667715934417893564045698440e-01 1.73692667715926923888147825891792e-01 9.13427474706147296545921676624857e-03
            4.10668319049825147892818222317146e-01 4.10668319049824981359364528543665e-01 1.78663361900349870747817249139189e-01 1.19292674837910006591101819140022e-02
            4.10668319049825536470876841121935e-01 1.78663361900355255329486681148410e-01 4.10668319049819263710787708987482e-01 1.19292674837906675922027943670400e-02
            1.78663361900354061839735209105129e-01 4.10668319049820540467266027917503e-01 4.10668319049825425448574378606281e-01 1.19292674837908913715311953751552e-02
            2.49556756982751537510267780817230e-01 2.49556756982739519346026213497680e-01 5.00886486034509026410432852571830e-01 1.22658688223783677445632989133628e-02
            2.49556756982747429685076667738031e-01 5.00886486034497702135581675975118e-01 2.49556756982754812668190425029024e-01 1.22658688223784249904380061479969e-02
            5.00886486034504141429124501883052e-01 2.49556756982744321060607717299717e-01 2.49556756982751537510267780817230e-01 1.22658688223781682513635615805470e-02
            9.80030809579556794197685576364165e-01 1.60168116844329466719187848866568e-02 3.95237873601025913039563874917803e-03 3.75945691771745985499536546470267e-04
            9.80030809579556794197685576364165e-01 3.95237873600955656738786814230480e-03 1.60168116844336475002030795167229e-02 3.75945691771704894237199345852218e-04
            1.60168116844335885196048963052817e-02 9.80030809579558348509920051583322e-01 3.95237873600806643992200406501070e-03 3.75945691771742841313236338507409e-04
            1.60168116844335885196048963052817e-02 3.95237873600896762876777401629624e-03 9.80030809579557460331500351458089e-01 3.75945691771595010347017939977832e-04
            3.95237873601129909711948684503113e-03 9.80030809579556794197685576364165e-01 1.60168116844319197156210066168569e-02 3.75945691771786534660787504336099e-04
            3.95237873600939697282807827605211e-03 1.60168116844335885196048963052817e-02 9.80030809579557016242290501395473e-01 3.75945691771573272093459605613930e-04
            9.56210240585889215125803275441285e-01 4.00653234502970218011697056681442e-02 3.72443596381376307302701889057062e-03 5.77435002003155294775238459692446e-04
            9.56210240585890103304222975566518e-01 3.72443596381326911051723449475048e-03 4.00653234502966262842171829561266e-02 5.77435002003019661083460523087751e-04
            4.00653234503037872227260152158124e-02 9.56210240585880777430816124251578e-01 3.72443596381544228535176443983801e-03 5.77435002003340476506298983849774e-04
            4.00653234502934343930213856310729e-02 3.72443596381445956450262357861902e-03 9.56210240585892101705667300848290e-01 5.77435002003157788440235176352644e-04
            3.72443596381283976645693023499462e-03 9.56210240585892323750272225879598e-01 4.00653234502947874773326475406066e-02 5.77435002003023781051715968004601e-04
            3.72443596381152484606213981521705e-03 4.00653234502949193163168217779457e-02 9.56210240585893544995599313551793e-01 5.77435002002882509508641106776849e-04
            9.18991377915119977970448417181615e-01 7.64894022503797427203764414116449e-02 4.51921983450027930917514140674029e-03 9.95858515622234001865575869771874e-04
            9.18991377915119755925843492150307e-01 4.51921983449780993030708842184140e-03 7.64894022503824350112111574162554e-02 9.95858515621874914106048670703331e-04
            7.64894022503812553992474931874312e-02 9.18991377915118534680516404478112e-01 4.51921983450020992023610233445652e-03 9.95858515622080261997517425243132e-04
            7.64894022503812553992474931874312e-02 4.51921983449856713710435229813811e-03 9.18991377915120200015053342212923e-01 9.95858515622115823828774949788567e-04
            4.51921983449772319413328958148668e-03 9.18991377915119755925843492150307e-01 7.64894022503824766445745808596257e-02 9.95858515621842388040874105570310e-04
            4.51921983450051002739744632208385e-03 7.64894022503812553992474931874312e-02 9.18991377915118201613609016931150e-01 9.95858515622329628497189091262953e-04
            8.69266446632026346108546022151131e-01 1.25877991283983525772072198378737e-01 4.85556208399012811938177947013173e-03 1.38275956888925547969004803405824e-03
            8.69266446632019906815003196243197e-01 4.85556208399166768646670888642802e-03 1.25877991283988438508956164696428e-01 1.38275956888958377610787664480085e-03
            1.25877991283981305326022948065656e-01 8.69266446632028344509990347432904e-01 4.85556208399029465283547324361280e-03 1.38275956888930470246867887595954e-03
            1.25877991283990742221732261896250e-01 4.85556208399150809190691902017534e-03 8.69266446632017797391256408445770e-01 1.38275956888958767923569759261682e-03
            4.85556208399082721294259812339078e-03 8.69266446632021017038027821399737e-01 1.25877991283988133197624392778380e-01 1.38275956888936173150295161349277e-03
            4.85556208399116288193519963556355e-03 1.25877991283984885795277364195499e-01 8.69266446632023903617891846806742e-01 1.38275956888947730745453856826543e-03
            8.08161929939749112961067112337332e-01 1.86984740735146592882642835320439e-01 4.85332932510429415629005234222859e-03 1.64348980314667930148486707508937e-03
            8.08161929939749557050276962399948e-01 4.85332932510483885946150905965624e-03 1.86984740735145593681920672679553e-01 1.64348980314673719788087780102614e-03
            1.86984740735145787970949982081947e-01 8.08161929939749446027974499884294e-01 4.85332932510473824549990240484476e-03 1.64348980314690720078152352812140e-03
            1.86984740735145787970949982081947e-01 4.85332932510453094604452317639698e-03 8.08161929939749668072579424915602e-01 1.64348980314650257653075193786663e-03
            4.85332932510489957478316824790454e-03 8.08161929939749557050276962399948e-01 1.86984740735145593681920672679553e-01 1.64348980314674066732782975464033e-03
            4.85332932510419874649887361783840e-03 1.86984740735145787970949982081947e-01 8.08161929939750001139486812462565e-01 1.64348980314661923668451137814372e-03
            9.31015414728788681664184423425468e-01 4.79634591905657992061406957873260e-02 2.10211260806455191296748807872063e-02 1.73634746626700553831512685576399e-03
            9.31015414728788681664184423425468e-01 2.10211260806391006528137666009570e-02 4.79634591905722176830018099735753e-02 1.73634746626682404287145278232174e-03
            4.79634591905629750763218055453763e-02 9.31015414728788681664184423425468e-01 2.10211260806483224428120593074709e-02 1.73634746626675487077284820713885e-03
            4.79634591905694421254402470822242e-02 2.10211260806433507253299097783383e-02 9.31015414728787238374252410721965e-01 1.73634746626730499495516735208867e-03
            2.10211260806381569632428352178977e-02 9.31015414728788681664184423425468e-01 4.79634591905732099448300687072333e-02 1.73634746626688389083137398216650e-03
            2.10211260806409117041226863875636e-02 4.79634591905722315607896177880320e-02 9.31015414728786905307345023175003e-01 1.73634746626720568203616767988251e-03
            7.38140363468471782937285752268508e-01 2.56995801448647809284153709086240e-01 4.86383508288040777856053864525165e-03 1.82964695987906933449806956559769e-03
            7.38140363468471893959588214784162e-01 4.86383508287979889062047078596152e-03 2.56995801448648308884514790406683e-01 1.82964695987896351636603498036493e-03
            2.56995801448652416709705903485883e-01 7.38140363468467120000582326611038e-01 4.86383508288046328971176990307868e-03 1.82964695987920529345049924785371e-03
            2.56995801448641814079820733240922e-01 4.86383508287964016342241890811238e-03 7.38140363468478555297735965723405e-01 1.82964695987875903583630421422868e-03
            4.86383508288007471165315109828953e-03 7.38140363468472560093402989878086e-01 2.56995801448647420706095090281451e-01 1.82964695987904981885896482651788e-03
            4.86383508288008511999400695913209e-03 2.56995801448649974219051728141494e-01 7.38140363468469895558143889502389e-01 1.82964695987906803345546258299237e-03
            6.61964863274298598305733776214765e-01 3.33118225448955973533315955137368e-01 4.91691127674542816095026864786632e-03 1.98073891497663450572686016926127e-03
            6.61964863274293047190610650432063e-01 4.91691127674512024753328276460707e-03 3.33118225448961857715346468467033e-01 1.98073891497660805119385152295308e-03
            3.33118225448962690382614937334438e-01 6.61964863274292047989888487791177e-01 4.91691127674531713864780613221228e-03 1.98073891497654213170176440428349e-03
            3.33118225448954308198779017402558e-01 4.91691127674512284961849672981771e-03 6.61964863274300596707178101496538e-01 1.98073891497670909883632717196633e-03
            4.91691127674503090927426995904170e-03 6.61964863274293269235215575463371e-01 3.33118225448961746693044005951378e-01 1.98073891497663016891817022724354e-03
            4.91691127674551576448580547662459e-03 3.33118225448962690382614937334438e-01 6.61964863274291825945283562759869e-01 1.98073891497675203324235759794192e-03
            5.81189175108866140639918285160093e-01 4.13846657841547505007895324524725e-01 4.96416704958635435218639031518251e-03 2.10067434347362526694813489314129e-03
            5.81189175108862587926239484659163e-01 4.96416704958642721057238134108047e-03 4.13846657841551002210422893767827e-01 2.10067434347346176926052407907264e-03
            4.13846657841549614431642112322152e-01 5.81189175108864031216171497362666e-01 4.96416704958635435218639031518251e-03 2.10067434347344875883445425301943e-03
            4.13846657841551113232725356283481e-01 4.96416704958651134466096621622455e-03 5.81189175108862365881634559627855e-01 2.10067434347372067673931361753148e-03
            4.96416704958626154448042555600296e-03 5.81189175108860256457887771830428e-01 4.13846657841553500212228300370043e-01 2.10067434347347044287790396310811e-03
            4.96416704958658420304695724212252e-03 4.13846657841552834078413525276119e-01 5.81189175108860589524795159377391e-01 2.10067434347358883775513938019230e-03
            8.84274577566934505412632461229805e-01 9.11814742788523008298895433654252e-02 2.45439481542131937574779954047699e-02 2.76209123159379312667516259693912e-03
            8.84274577566933284167305373557610e-01 2.45439481542091414434381135833974e-02 9.11814742788575743892565128589922e-02 2.76209123159357498519805851344699e-03
            9.11814742788418924890336825228587e-02 8.84274577566945496620576250279555e-01 2.45439481542125692570266437542159e-02 2.76209123159351470355726831940046e-03
            9.11814742788630977488040230127808e-02 2.45439481542108310641037149935073e-02 8.84274577566926067717645310040098e-01 2.76209123159412922934863310331366e-03
            2.45439481542071430419937883016246e-02 8.84274577566932062921978285885416e-01 9.11814742788608079138157336274162e-02 2.76209123159338286457309408206129e-03
            2.45439481542168436156714506068965e-02 9.11814742788630838710162151983241e-02 8.84274577566920072513312334194779e-01 2.76209123159424805790673751459963e-03
            8.27639190276182601380128289747518e-01 1.46943421195835172143873137429182e-01 2.54173885279822264759985728233005e-02 3.46900940527512238395480004271576e-03
            8.27639190276168834614622937806416e-01 2.54173885279898627287398227281301e-02 1.46943421195841306126084191419068e-01 3.46900940527575859378961453671764e-03
            1.46943421195824069913626885863778e-01 8.27639190276191705208930216031149e-01 2.54173885279842526330185137339868e-02 3.46900940527552050299253671994393e-03
            1.46943421195851520177910742859240e-01 2.54173885279888080168664288294167e-02 8.27639190276159619763518549007131e-01 3.46900940527570611840446623830303e-03
            2.54173885279870385989209324861804e-02 8.27639190276168834614622937806416e-01 1.46943421195844137194796985568246e-01 3.46900940527546629288391244472223e-03
            2.54173885279852795893162920037867e-02 1.46943421195844137194796985568246e-01 8.27639190276170610971462338056881e-01 3.46900940527510330199656429783772e-03
            7.62587568566484019783047187956981e-01 2.12066839505367032980842623146600e-01 2.53455919281489472361101888964185e-02 3.92705197335976429612314220207736e-03
            7.62587568566486018184491513238754e-01 2.53455919281477641546995727139802e-02 2.12066839505366228069149769908108e-01 3.92705197335930372704027035979379e-03
            2.12066839505360871243055953527801e-01 7.62587568566489015786658001161413e-01 2.53455919281501129702860453107860e-02 3.92705197336029772359200507025889e-03
            2.12066839505372944918448752105178e-01 2.53455919281464422954108783869742e-02 7.62587568566480578091670849971706e-01 3.92705197335870177799410640773203e-03
            2.53455919281491068306699787626712e-02 7.62587568566481244225485625065630e-01 2.12066839505369642004950492264470e-01 3.92705197335942862713054068990459e-03
            2.53455919281468967929615843104330e-02 2.12066839505368087692716017045313e-01 7.62587568566485018983769350597868e-01 3.92705197335912505052224474866307e-03
            6.90118710560341575011022996477550e-01 2.84342556304586102644549328033463e-01 2.55387331350723223444276754889870e-02 4.32349605661401081152517278383129e-03
            6.90118710560342352167140234087128e-01 2.55387331350694947451618332934231e-02 2.84342556304588156557144884573063e-01 4.32349605661407499629378392569379e-03
            2.84342556304586491222607946838252e-01 6.90118710560341575011022996477550e-01 2.55387331350719337663690566841979e-02 4.32349605661426234642918942085998e-03
            2.84342556304586491222607946838252e-01 2.55387331350689604503312324368380e-02 6.90118710560344572613189484400209e-01 4.32349605661357105912401266323286e-03
            2.55387331350701574095296564337332e-02 6.90118710560342352167140234087128e-01 2.84342556304587490423330109479139e-01 4.32349605661383647181583711471831e-03
            2.55387331350716770272946121167479e-02 2.84342556304585492021885784197366e-01 6.90118710560342796256350084149744e-01 4.32349605661407933310247386771152e-03
            8.38318567454824137286095719900914e-01 1.01421462494415193233976424380671e-01 6.02599700507606694799278557184152e-02 4.55352437922805237252577015283350e-03
            8.38318567454823804219188332353951e-01 6.02599700507590388398604375197465e-02 1.01421462494417163879845134033530e-01 4.55352437922780170498349150420836e-03
            1.01421462494409919674609454887104e-01 8.38318567454828911245101608074037e-01 6.02599700507612245914401682966854e-02 4.55352437922813043508218910915275e-03
            1.01421462494417274902147596549185e-01 6.02599700507605237631558736666193e-02 8.38318567454822138884651394619141e-01 4.55352437922795696273459142844331e-03
            6.02599700507544522309899548417889e-02 8.38318567454823804219188332353951e-01 1.01421462494421743549821712804260e-01 4.55352437922765165140281951039469e-03
            6.02599700507679761352086700298969e-02 1.01421462494414804655917805575882e-01 8.38318567454817253903343043930363e-01 4.55352437922779129664263564336579e-03
            6.11809416665966332438131303206319e-01 3.62362307698882724071864913639729e-01 2.58282756351509434900037831539521e-02 4.67622936259775960621265511463207e-03
            6.11809416665957672698539226985304e-01 2.58282756351504681757713655088082e-02 3.62362307698891883411818071181187e-01 4.67622936259786802642990366507547e-03
            3.62362307698885832696333864078042e-01 6.11809416665963778925174665346276e-01 2.58282756351504438896427018335089e-02 4.67622936259767720684754621629509e-03
            3.62362307698886387807846176656312e-01 2.58282756351506832814823866328879e-02 6.11809416665962890746754965221044e-01 4.67622936259789404728204331718189e-03
            2.58282756351497777558279267395847e-02 6.11809416665963112791359890252352e-01 3.62362307698887109452812183008064e-01 4.67622936259765899225104845982059e-03
            2.58282756351520051407710809598939e-02 3.62362307698885166562519088984118e-01 6.11809416665962890746754965221044e-01 4.67622936259807445852354490511971e-03
            5.29148852224490484630337050475646e-01 4.44835853963695238011410992839956e-01 2.60152938118142773582519566843985e-02 4.90579532447948506862500650527181e-03
            5.29148852224484156359096687083365e-01 2.60152938118148047141886536337552e-02 4.44835853963701066682290274911793e-01 4.90579532447916674686716476116999e-03
            4.44835853963697180901704086863901e-01 5.29148852224488708273497650225181e-01 2.60152938118141108247982629109174e-02 4.90579532447938358730166186205679e-03
            4.44835853963697180901704086863901e-01 2.60152938118150475754752903867484e-02 5.29148852224487820095077950099949e-01 4.90579532447930812683045687094818e-03
            2.60152938118142808276989086380127e-02 5.29148852224488042139682875131257e-01 4.44835853963697624990913936926518e-01 4.90579532447929338168091106808788e-03
            2.60152938118152418645045997891430e-02 4.44835853963697180901704086863901e-01 5.29148852224487598050473025068641e-01 4.90579532447949981377455230813212e-03
            7.78589579815803345397284829232376e-01 1.59964054253392773397024484438589e-01 6.14463659308038812056906863290351e-02 5.49383485445390732510029252466666e-03
            7.78589579815792576233945965213934e-01 6.14463659308165585648531248352811e-02 1.59964054253390858262307006043557e-01 5.49383485445371216870924513386854e-03
            1.59964054253388304749350368183514e-01 7.78589579815803345397284829232376e-01 6.14463659308083220977891869551968e-02 5.49383485445419962600599461666206e-03
            1.59964054253390774995580159156816e-01 6.14463659308144352633185292233975e-02 7.78589579815794796679995215527015e-01 5.49383485445359073806592675737193e-03
            6.14463659308128323788267266536423e-02 7.78589579815790133743291789869545e-01 1.59964054253396992244518060033442e-01 5.49383485445379196598914006699488e-03
            6.14463659308071563636133305408293e-02 1.59964054253384030390705561330833e-01 7.78589579815808896512407955015078e-01 5.49383485445322818085944760468919e-03
            7.10623785618011427622775499912677e-01 2.27893984493793722023724512837362e-01 6.14822298881948503534999872499611e-02 6.28606573294208676588867845680397e-03
            7.10623785618027414834330102166859e-01 6.14822298881887718824401645179023e-02 2.27893984493783813283229733315238e-01 6.28606573294147007169296870188191e-03
            2.27893984493788864797991777777497e-01 7.10623785618015091358756762929261e-01 6.14822298881960715988270749221556e-02 6.28606573294211712354950805092813e-03
            2.27893984493783979816683427088719e-01 6.14822298881861836750140071217174e-02 7.10623785618029857324984277511248e-01 6.28606573294132175283577268487534e-03
            6.14822298881921996960286946887209e-02 7.10623785618016312604083850601455e-01 2.27893984493791501577675262524281e-01 6.28606573294153512382331783214795e-03
            6.14822298881903331335685436442873e-02 2.27893984493780205058399701556482e-01 7.10623785618029524258076889964286e-01 6.28606573294195492690450421946480e-03
            6.34277654844062399597248713689623e-01 3.03562715120037796001639662790694e-01 6.21596300358998044011116235196823e-02 6.94363231297635916799970701163147e-03
            6.34277654844071392403748177457601e-01 6.21596300358955855536180479248287e-02 3.03562715120033022042633774617570e-01 6.94363231297665233626714709203043e-03
            3.03562715120033077553785005875397e-01 6.34277654844068061734674301987980e-01 6.21596300358988607115406921366230e-02 6.94363231297632273880671149868249e-03
            3.03562715120035075955229331157170e-01 6.21596300358951137088325822332990e-02 6.34277654844069838091513702238444e-01 6.94363231297691601423549556670878e-03
            6.21596300358957104537083182549395e-02 6.34277654844067839690069376956671e-01 3.03562715120036408222858881345019e-01 6.94363231297624033944160260034550e-03
            6.21596300359004150237751673557796e-02 3.03562715120029968929316055437084e-01 6.34277654844069616046908777207136e-01 6.94363231297713458939346864440267e-03
            5.52626687887673773147412248363253e-01 3.84560072995418278374302190059097e-01 6.28132391169079484782855615776498e-02 7.37828444684584215662592754370053e-03
            5.52626687887675771548856573645025e-01 6.28132391169085035897978741559200e-02 3.84560072995415724861345552199055e-01 7.37828444684579705381555214671607e-03
            3.84560072995413726459901226917282e-01 5.52626687887679213240232911630301e-01 6.28132391169070602998658614524174e-02 7.37828444684579445173033818150543e-03
            3.84560072995421997621434684333508e-01 6.28132391169091419680370336209307e-02 5.52626687887668888166103897674475e-01 7.37828444684624374511061617454288e-03
            6.28132391169068243774731286066526e-02 5.52626687887675771548856573645025e-01 3.84560072995417390195882489933865e-01 7.37828444684571638917391922518618e-03
            6.28132391169106824024837010256306e-02 3.84560072995415613839043089683400e-01 5.52626687887673662125109785847599e-01 7.37828444684636344103045857423240e-03
            7.16871672741223320279857489367714e-01 1.71646103407856243094897763512563e-01 1.11482223850920436625244747119723e-01 7.49645973918343019870569321483345e-03
            7.16871672741206555912185649503954e-01 1.11482223850932246622669197222422e-01 1.71646103407861183587357345459168e-01 7.49645973918389076778856505711701e-03
            1.71646103407851691180496800370747e-01 7.16871672741222210056832864211174e-01 1.11482223850926098762670335418079e-01 7.49645973918383612399907178769354e-03
            1.71646103407866429391148699323821e-01 1.11482223850928902075807513938344e-01 7.16871672741204668533043786737835e-01 7.49645973918313963252346709964513e-03
            1.11482223850929831887590637506946e-01 7.16871672741206555912185649503954e-01 1.71646103407863570566860289545730e-01 7.49645973918399658592059964234977e-03
            1.11482223850921685626147450420831e-01 1.71646103407862293810381970615708e-01 7.16871672741216103830197425850201e-01 7.49645973918365050858714226933444e-03
            6.45570482899384456842994950420689e-01 2.42254408443599883460350952191220e-01 1.12175108657015659696654097388091e-01 8.52180142423126542328137844606317e-03
            6.45570482899385900132926963124191e-01 1.12175108657004585221983461451600e-01 2.42254408443609514645089575424208e-01 8.52180142423157593878357829453307e-03
            2.42254408443599161815384945839469e-01 6.45570482899384456842994950420689e-01 1.12175108657016409097195719368756e-01 8.52180142423183267785802286198305e-03
            2.42254408443608210133035640865273e-01 1.12175108657001171286182739095239e-01 6.45570482899390563069630388781661e-01 8.52180142423073633262120551989938e-03
            1.12175108657009095503021001150046e-01 6.45570482899379571861686599731911e-01 2.42254408443611346513080206932500e-01 8.52180142423135215945517728641789e-03
            1.12175108657011149415616557689646e-01 2.42254408443608210133035640865273e-01 6.45570482899380571062408762372797e-01 8.52180142423166094023390115808070e-03
            5.66843738230976668113214600452920e-01 3.19581218448544313037729125426267e-01 1.13575043320479018849056274120812e-01 9.26229049718121401346682119992693e-03
            5.66843738230963123392314173543127e-01 1.13575043320476631869553330034250e-01 3.19581218448560244738132496422622e-01 9.26229049718178820693736952307518e-03
            3.19581218448548531885222701021121e-01 5.66843738230973670511048112530261e-01 1.13575043320477853114880417706445e-01 9.26229049718180555417212929114612e-03
            3.19581218448554693623009370639920e-01 1.13575043320476451458311828446313e-01 5.66843738230968896552042224357137e-01 9.26229049718152626369249702520392e-03
            1.13575043320475035923955431371724e-01 5.66843738230966565083690511528403e-01 3.19581218448558357358990633656504e-01 9.26229049718130421908757199389584e-03
            1.13575043320482543807159458992828e-01 3.19581218448554749134160601897747e-01 5.66843738230962790325406785996165e-01 9.26229049718152452896902104839683e-03
            4.84400341724715965963099506552680e-01 4.01102153748802869870360154891387e-01 1.14497504526481108655389107298106e-01 9.65530625783643514647991423771600e-03
            4.84400341724701422041476917002001e-01 1.14497504526483842579587246746087e-01 4.01102153748814804767874875324196e-01 9.65530625783549839580288676188502e-03
            4.01102153748810419386927605955862e-01 4.84400341724708916046893136808649e-01 1.14497504526480664566179257235490e-01 9.65530625783668841610740685155179e-03
            4.01102153748808587518936974447570e-01 1.14497504526484258913221481179789e-01 4.84400341724707084178902505300357e-01 9.65530625783548972218550687784955e-03
            1.14497504526481871933718537093227e-01 4.84400341724705918444726648885990e-01 4.01102153748812195743767006206326e-01 9.65530625783565278619224869771642e-03
            1.14497504526485008313763103160454e-01 4.01102153748809087119298055768013e-01 4.84400341724705918444726648885990e-01 9.65530625783567013342700846578737e-03
            5.78477356973122636141226848849328e-01 2.45951501392227434017456744186347e-01 1.75571141634649929841316406964324e-01 1.05623976932274540180278776801970e-02
            5.78477356973136624951337125821738e-01 1.75571141634637689632469914613466e-01 2.45951501392225685416192959564796e-01 1.05623976932283075019780582692874e-02
            2.45951501392227489528607975444174e-01 5.78477356973121414895899761177134e-01 1.75571141634651040064341032120865e-01 1.05623976932277749418709333895094e-02
            2.45951501392227489528607975444174e-01 1.75571141634633887118610573452315e-01 5.78477356973138512330478988587856e-01 1.05623976932285330160299352542097e-02
            1.75571141634641991746690337095060e-01 5.78477356973128076234047512116376e-01 2.45951501392229987530413382046390e-01 1.05623976932281149476722248437000e-02
            1.75571141634647015505876765928406e-01 2.45951501392224075592807253087813e-01 5.78477356973128853390164749725955e-01 1.05623976932279241280898673949196e-02
            4.96282829301863182003273777809227e-01 3.25942627323929146054837246992975e-01 1.77774543374207671941888975197799e-01 1.15638068495018363213144851897596e-02
            4.96282829301863182003273777809227e-01 1.77774543374208421342430597178463e-01 3.25942627323928424409871240641223e-01 1.15638068495020774478776459659457e-02
            3.25942627323929479121744634539937e-01 4.96282829301863182003273777809227e-01 1.77774543374207394386132818908663e-01 1.15638068495020566311959342442606e-02
            3.25942627323933198368877128814347e-01 1.77774543374208088275523209631501e-01 4.96282829301858741111175277183065e-01 1.15638068495023307175051385797815e-02
            1.77774543374205951096200806205161e-01 4.96282829301860906046073296238319e-01 3.25942627323933142857725897556520e-01 1.15638068495022283688200559481629e-02
            1.77774543374213112034709638464847e-01 3.25942627323925926408065834039007e-01 4.96282829301861017068375758753973e-01 1.15638068495019716297456113807129e-02
            4.16568600024706914286554138016072e-01 3.31203743439009878901657657479518e-01 2.52227656536283262322939435762237e-01 1.32067902710617367939560651279862e-02
            4.16568600024701640727187168522505e-01 2.52227656536288813438062561544939e-01 3.31203743439009545834750269932556e-01 1.32067902710612684186175513900707e-02
            3.31203743439012154858858139050426e-01 4.16568600024703750150933956319932e-01 2.52227656536284094990207904629642e-01 1.32067902710618721023871913189396e-02
            3.31203743439011766280799520245637e-01 2.52227656536287758726189167646226e-01 4.16568600024700419481860080850311e-01 1.32067902710613430117270183927758e-02
            2.52227656536286315436257154942723e-01 4.16568600024698143524659599279403e-01 3.31203743439015541039083245777874e-01 1.32067902710617263856152092671437e-02
            2.52227656536289701616482261670171e-01 3.31203743439003883697324681634200e-01 4.16568600024706359175041825437802e-01 1.32067902710616066896953668674541e-02
            ];
        
    case  32
        
        % ALG. DEG.:   32
        % PTS CARD.:  210
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.8e-16
        
        xyw_bar=[
            3.19145350604780860329090508287209e-03 3.19145350623165069414710615092190e-03 9.93617092987720607055734944879077e-01 1.43085923909148856085146728212010e-04
            3.19145350611156012471392173779350e-03 9.93617092987746253207603785995161e-01 3.19145350614213896278670290485024e-03 1.43085923907762839132895571481185e-04
            9.93617092987416850036197502049617e-01 3.19145350614327000249303978307580e-03 3.19145350643987996130945816730673e-03 1.43085923919664858216735381368778e-04
            1.64655689217531008006201176385730e-02 1.64655689217531008006201176385730e-02 9.67068862156493791459865860815626e-01 5.54996635606638376997379324961912e-04
            1.64655689225811190101733672008777e-02 9.67068862156493791459865860815626e-01 1.64655689209250999383016278443392e-02 5.54996635627037315612042345236432e-04
            9.67068862156493791459865860815626e-01 1.64655689227284109110716059376500e-02 1.64655689207777976290625332467243e-02 5.54996635632184457005700028986439e-04
            4.98495116352493869427142669792374e-02 4.98495116355436237998155490913632e-02 9.00300976729206947624106760486029e-01 2.84820867498443257292972674576959e-03
            4.98495116350614331235391318841721e-02 9.00300976729200952419773784640711e-01 4.98495116357375867011114678462036e-02 2.84820867496734554669135519588963e-03
            9.00300976729124347031074648839422e-01 4.98495116353910652784442447682522e-02 4.98495116354845876904811063923262e-02 2.84820867498196102565732878986182e-03
            4.88219324360620376968000755368848e-01 4.88219324361344630958114976237994e-01 2.35613512780350475850354996509850e-02 4.23536581735520917257886708284786e-03
            4.88219324361224116248791915495531e-01 2.35613512779833944588148142429418e-02 4.88219324360792517047968885890441e-01 4.23536581733655308895647451095101e-03
            2.35613512780159378712241391440330e-02 4.88219324361224116248791915495531e-01 4.88219324360759932002196137545980e-01 4.23536581734212328603783603853117e-03
            9.41148599404452168570855974394362e-02 9.41148599402267110880515588178241e-02 8.11770280119328058177075035928283e-01 4.64365914460811417957541991086146e-03
            9.41148599404054153616527855774621e-02 8.11770280118929044022024754667655e-01 9.41148599406655961274736910127103e-02 4.64365914464898599939290946281290e-03
            8.11770280119172182864417663950007e-01 9.41148599406814168055745994934114e-02 9.41148599401464003300077365565812e-02 4.64365914461504786930889920881782e-03
            1.56836966604454702833848500631575e-01 1.56836966605318151035675100501976e-01 6.86326066790227118374900783237536e-01 7.75487814089327771871484884513848e-03
            1.56836966604974537009553614552715e-01 6.86326066790763578140399658877868e-01 1.56836966604261940361197957827244e-01 7.75487814084894859500973751664787e-03
            6.86326066790314603949241245572921e-01 1.56836966604918914836019894210040e-01 1.56836966604766481214738860217039e-01 7.75487814088207053769830068290503e-03
            4.47917974916116512495989354647463e-01 4.47917974915863603690979743987555e-01 1.04164050168019883813030901364982e-01 8.87199984566272234487449566131545e-03
            4.47917974916230809956374514513300e-01 1.04164050167794647316910072731844e-01 4.47917974915974625993442259641597e-01 8.87199984567290170223152756534546e-03
            1.04164050167938004864964796070126e-01 4.47917974915831462734416845705709e-01 4.47917974916230587911769589481992e-01 8.87199984566704354105315388778763e-03
            2.27359360510421992795926371400128e-01 2.27359360510134639321577765258553e-01 5.45281278979443340126920247712405e-01 1.03947593390126911266913012354962e-02
            2.27359360510365593466275413447875e-01 5.45281278978972272497571793792304e-01 2.27359360510662078525001561501995e-01 1.03947593390556151243808713502403e-02
            5.45281278979131367457000578724546e-01 2.27359360510439229008383676955418e-01 2.27359360510429403534615744320035e-01 1.03947593390289506898316318483921e-02
            3.84316624257439620748755260137841e-01 3.84316624257142525067365568247624e-01 2.31366751485417854183879171614535e-01 1.19592550228468236239720567937184e-02
            3.84316624257420691446185401218827e-01 2.31366751484876870259554948461300e-01 3.84316624257702410538684034690959e-01 1.19592550228443082749318904234315e-02
            2.31366751485209160010825257813849e-01 3.84316624257017791510548931910307e-01 3.84316624257773076234201425904757e-01 1.19592550228466883155409306027650e-02
            3.06547093375159840356047880050028e-01 3.06547093374228252216084911196958e-01 3.86905813250611907427867208753014e-01 1.24933120762573297846120823351157e-02
            3.06547093374736845383665695408126e-01 3.86905813249718177893043957737973e-01 3.06547093375544976723290346853901e-01 1.24933120762627004884937065298800e-02
            3.86905813250220442789384378556861e-01 3.06547093374826551404055408056593e-01 3.06547093374953005806560213386547e-01 1.24933120762589257302099809976426e-02
            9.78394604898897179978689564450178e-01 1.78715560665132847106839619755192e-02 3.73383903458953531062647357430251e-03 3.65304520928884715862822174514690e-04
            9.78394604897416142463839605625253e-01 3.73383903520391842098535839511442e-03 1.78715560673799386814941669854306e-02 3.65304520993771884851408771055503e-04
            1.78715560663958682174889958105268e-02 9.78394604898897179978689564450178e-01 3.73383903470692057879887215676717e-03 3.65304520937760807998417877939801e-04
            1.78715560663392850071495843167213e-02 3.73383903487901165282880633355944e-03 9.78394604898781716784128548169974e-01 3.65304520960409357700771071364443e-04
            3.73383903506577675274202832156334e-03 9.78394604898398134729120556585258e-01 1.78715560665361206105217206641100e-02 3.65304520979064194490665284931197e-04
            3.73383903503422213271401020051599e-03 1.78715560667996875565677328268066e-02 9.78394604898166098116973898868309e-01 3.65304520971560268624567235917766e-04
            9.57376599458149901700210193666862e-01 3.94715791576244062754952324212354e-02 3.15182138422569202429457391190226e-03 3.72432834574188634940705711073861e-04
            9.57376599458149901700210193666862e-01 3.15182138377239143278707445006148e-03 3.94715791580777103364496838366904e-02 3.72432834520339023538781919597795e-04
            3.94715791577081517860214887605252e-02 9.57376599458149901700210193666862e-01 3.15182138414193957487441366538405e-03 3.72432834572970425379700998291810e-04
            3.94715791578275077000625969958492e-02 3.15182138415708978235185711014310e-03 9.57376599458015453691928087209817e-01 3.72432834574222570468704507362645e-04
            3.15182138402951127903461880919167e-03 9.57376599458149901700210193666862e-01 3.94715791578206243173099210252985e-02 3.72432834558825815417237858895305e-04
            3.15182138379868290178897893838439e-03 3.94715791578275077000625969958492e-02 9.57376599458373833684277087741066e-01 3.72432834533442528365115853361544e-04
            9.29763016779499351649462823843351e-01 6.62735149105910109623707171522256e-02 3.96346830990963738816645900442381e-03 7.06032907779036394795013631409120e-04
            9.29763016779499351649462823843351e-01 3.96346830996999622415133401887033e-03 6.62735149105306564631945320797968e-02 7.06032907785073991432933659950777e-04
            6.62735149105613263742498020292260e-02 9.29763016779510564902011537924409e-01 3.96346830992810872373866004636511e-03 7.06032907768598997320930177323817e-04
            6.62735149105613263742498020292260e-02 3.96346830994368046902076585524810e-03 9.29763016779495021779666785732843e-01 7.06032907772308595054089330744773e-04
            3.96346830989261888350538853842409e-03 9.29763016779499351649462823843351e-01 6.62735149106080667635865211195778e-02 7.06032907766412812060330406183084e-04
            3.96346830995000353609070131710723e-03 6.62735149105613263742498020292260e-02 9.29763016779488693508426422340563e-01 7.06032907778226604192384208147359e-04
            8.85614537945472846125483101786813e-01 1.10207799879526918696015513887687e-01 4.17766217500023517850138432550011e-03 1.11236381582007718554772068841885e-03
            8.85614537945883739666896872222424e-01 4.17766217511278056817580761617137e-03 1.10207799879003476295480368207791e-01 1.11236381583268905889894107019700e-03
            1.10207799878525483650015814873768e-01 8.85614537946382562871860955056036e-01 4.17766217509196735591103788465261e-03 1.11236381582066720836998729993184e-03
            1.10207799879031217993308189306845e-01 4.17766217501465420003370354606886e-03 8.85614537945954127806658107147086e-01 1.11236381581691001416145603286623e-03
            4.17766217500311135002455387166265e-03 8.85614537945945135000158643379109e-01 1.10207799879051715485900331259472e-01 1.11236381581109934103823722040261e-03
            4.17766217508616297116041948811471e-03 1.10207799878877507615548836383823e-01 8.85614537946036284310480368731078e-01 1.11236381582495262587695350475769e-03
            8.27604925831065152408427820773795e-01 1.68070571856776151964396603943896e-01 4.32450231215869562717557528230827e-03 1.39183250066186696425329927251369e-03
            8.27604925832203464075575993774692e-01 4.32450231215218087316154438326521e-03 1.68070571855644362857518103737675e-01 1.39183250065465940505105013613729e-03
            1.68070571855081146717125761824718e-01 8.27604925832789106721065763849765e-01 4.32450231212977431738408995443024e-03 1.39183250064947475026222445393387e-03
            1.68070571856045070102680938362028e-01 4.32450231217714354398085063735380e-03 8.27604925831777693545632246241439e-01 1.39183250066480797106638345184138e-03
            4.32450231217312765913396432893023e-03 8.27604925831887383580465211707633e-01 1.68070571855939543404190317232860e-01 1.39183250066194567733102172013560e-03
            4.32450231213263053958728576731119e-03 1.68070571855577416409133206798288e-01 8.27604925832289950449194293469191e-01 1.39183250065122508624948505229213e-03
            9.40702526955874196445961388235446e-01 4.04066851816024183441378170300595e-02 1.88907878625233852099007947344944e-02 1.41436644368965816487282438629336e-03
            9.40702526956985640715913632448064e-01 1.88907878617686902922301328544563e-02 4.04066851812456689918562346974795e-02 1.41436644372262463296985313121468e-03
            4.04066851817778405209224956706748e-02 9.40702526955874196445961388235446e-01 1.88907878623479907886917317227926e-02 1.41436644368558178154471338672238e-03
            4.04066851806516164069549290616123e-02 1.88907878624055107497081706924291e-02 9.40702526956942897129465563921258e-01 1.41436644369187622567729523126445e-03
            1.88907878620161624738660322009309e-02 9.40702526957222451287066178338137e-01 4.04066851807614035863025492290035e-02 1.41436644370087141742153846735164e-03
            1.88907878617485813776966097066179e-02 4.04066851815257505053935460637149e-02 9.40702526956725626483546420786297e-01 1.41436644369511733965172339821947e-03
            7.61456947234849690353541973308893e-01 2.34130448559401543873192963474139e-01 4.41260420574876577326506321696797e-03 1.55161657662295057909240814097984e-03
            7.61456947235687353625621653918643e-01 4.41260420569694264414373208182951e-03 2.34130448558615700260787662045914e-01 1.55161657662475339046481703775271e-03
            2.34130448558233172917653064359911e-01 7.61456947236077374974172471411293e-01 4.41260420568945210817446422879584e-03 1.55161657662728283413322571959725e-03
            2.34130448558700909877927642810391e-01 4.41260420575276778032414171093478e-03 7.61456947235546355301494259038009e-01 1.55161657662932026685576047952964e-03
            4.41260420575270793236422051109002e-03 7.61456947235597203516022091207560e-01 2.34130448558650061663399810640840e-01 1.55161657663377525358250341724897e-03
            4.41260420568741727753714343407410e-03 2.34130448558700909877927642810391e-01 7.61456947235611636415342218242586e-01 1.55161657662481193738213125499215e-03
            6.91098245011027567663575155165745e-01 3.04467727385770547954280118574388e-01 4.43402760320188438214472625986673e-03 1.63602214652752275120262126506532e-03
            6.91098245010800527055039310653228e-01 4.43402760321616289107549135906083e-03 3.04467727385983322196949529825361e-01 1.63602214655195611452132009588922e-03
            3.04467727385771491643851049957448e-01 6.91098245011001255377891538955737e-01 4.43402760322730848940864234464243e-03 1.63602214655811741862712072048680e-03
            3.04467727385450193100524529654649e-01 4.43402760318876033168722372579396e-03 6.91098245011361078660172552190488e-01 1.63602214653277657809005152245163e-03
            4.43402760319638097191718983935971e-03 6.91098245011091405487491101666819e-01 3.04467727385712261245487297856016e-01 1.63602214654403580081087898889791e-03
            4.43402760321932529197219707839395e-03 3.04467727386134423550601013630512e-01 6.91098245010646206054616413894109e-01 1.63602214655226576266178195595558e-03
            6.16960568882195192763617797027109e-01 3.78576515897221910211811746194144e-01 4.46291522058289702457045677874703e-03 1.74911403514299199600523504471994e-03
            6.16960568881680049280191724392353e-01 4.46291522062344792054489062138600e-03 3.78576515897696475043687769357348e-01 1.74911403515275783865368097735882e-03
            3.78576515897503240726251760861487e-01 6.16960568881862014833927787549328e-01 4.46291522063479995097168284701183e-03 1.74911403515672103127498449026689e-03
            3.78576515897201648641612337087281e-01 4.46291522057206888063340954886371e-03 6.16960568882226279008307301410241e-01 1.74911403514115037019505116688833e-03
            4.46291522058339315548458614557603e-03 6.16960568881680049280191724392353e-01 3.78576515897736554094876737508457e-01 1.74911403514833581167298159897427e-03
            4.46291522062590081953992182661750e-03 3.78576515897704413138313839226612e-01 6.16960568881669724206062710436527e-01 1.74911403514845203814587204504960e-03
            5.38216147161096669648827628407162e-01 4.57285169383518652619358135780203e-01 4.49868345538467773181423581263516e-03 1.85624168263086287110341654482681e-03
            5.38216147160986979613994662940968e-01 4.49868345537645947934679568902538e-03 4.57285169383636558304573327404796e-01 1.85624168262024311082392102889571e-03
            4.57285169383451983726729395129951e-01 5.38216147161161395651163275033468e-01 4.49868345538662062210732983658090e-03 1.85624168262580875425615811735724e-03
            4.57285169383451983726729395129951e-01 4.49868345537558951552359332026754e-03 5.38216147161172386859107064083219e-01 1.85624168262405321409846958857770e-03
            4.49868345538401940425510261434283e-03 5.38216147160670788096581418358255e-01 4.57285169383945144794267889665207e-01 1.85624168262347533434053481471437e-03
            4.49868345538044934334154234534253e-03 4.57285169383377654295230740899569e-01 5.38216147161241886820448598882649e-01 1.85624168261553268606534050633172e-03
            8.99401392370284491484255795512581e-01 7.94579312680868260487798693247896e-02 2.11406763616286824669643351626291e-02 2.20979001484498949192669670082978e-03
            8.99401392369924779224277244793484e-01 2.11406763620120320690265458551949e-02 7.94579312680631921761431613049353e-02 2.20979001486623768610306761672746e-03
            7.94579312677945598375472968655231e-02 8.99401392370285601707280420669122e-01 2.11406763619198523329600902798120e-02 2.20979001484416810036082168267058e-03
            7.94579312671745419116575703810668e-02 2.11406763616783927028919265467266e-02 8.99401392371147023752087079628836e-01 2.20979001484782489744818079202560e-03
            2.11406763615650632182063617392487e-02 8.99401392370791197272694716957631e-01 7.94579312676437776730153927928768e-02 2.20979001484918318592987063198052e-03
            2.11406763619169414669674011975076e-02 7.94579312680107063826540070294868e-02 8.99401392370072327864249928097706e-01 2.20979001484999330179315180089361e-03
            8.45458162389810419234947858058149e-01 1.32357395969260355217400615401857e-01 2.21844416409292255476515265399939e-02 2.91512442538648065479267579291900e-03
            8.45458162389776668454999253299320e-01 2.21844416412726834486601745766166e-02 1.32357395968950658504681427984906e-01 2.91512442540139624091999337451853e-03
            1.32357395968531993402095281453512e-01 8.45458162390312684131288278877037e-01 2.21844416411552947110408240405377e-02 2.91512442538691563670427697729792e-03
            1.32357395968369484506865774164908e-01 2.21844416410163190744864181169760e-02 8.45458162390614220704776471393416e-01 2.91512442540032895230139864395369e-03
            2.21844416409958632152576996077187e-02 8.45458162390093193039319885428995e-01 1.32357395968910940275975463009672e-01 2.91512442539931760851490416541765e-03
            2.21844416411478527473288835381027e-02 1.32357395968795726881594987389690e-01 8.45458162390056444657204792747507e-01 2.91512442538453429505262981535907e-03
            7.82095750515384047218958585290238e-01 1.95018153177622444172101268122788e-01 2.28860963069935086089401465869742e-02 3.40299157382555061968076870471123e-03
            7.82095750516188403800299511203775e-01 2.28860963067741354781681195618148e-02 1.95018153177037467660426273141638e-01 3.40299157380277196571771725075450e-03
            1.95018153176744118981744691154745e-01 7.82095750516571985855307502788492e-01 2.28860963066839229185234216856770e-02 3.40299157380039365983215304822807e-03
            1.95018153176870295828493340195564e-01 2.28860963070662316865000462939861e-02 7.82095750516063392687726718577323e-01 3.40299157384913982318797032178281e-03
            2.28860963070526349238953400799801e-02 7.82095750515546139780553858145140e-01 1.95018153177401232234444705682108e-01 3.40299157384416246785452386802717e-03
            2.28860963066924646969191314838099e-02 1.95018153176893555000859237225086e-01 7.82095750516414001118903343012789e-01 3.40299157379233023143494385465146e-03
            7.13726302839667536836998351645889e-01 2.63119135268269244143368723598542e-01 2.31545618920632190196329247555695e-02 3.62237822460893896941991343396694e-03
            7.13726302839667536836998351645889e-01 2.31545618919372885036178644213578e-02 2.63119135268395198945512447608053e-01 3.62237822464897638724545814170597e-03
            2.63119135268292392293432158112410e-01 7.13726302839753135032196951215155e-01 2.31545618919544171632196594146080e-02 3.62237822466000445806311169860692e-03
            2.63119135267934012301083157581161e-01 2.31545618920304466237247709159419e-02 7.13726302840035464747359128523385e-01 3.62237822462996858843831127217072e-03
            2.31545618920493134762494946699007e-02 7.13726302839617243733982832054608e-01 2.63119135268333415034192057646578e-01 3.62237822462047574789689718954833e-03
            2.31545618919266234236875590113414e-02 2.63119135268292392293432158112410e-01 7.13726302839780890607812580128666e-01 3.62237822465381583206256443929760e-03
            8.53648252343191416002810001373291e-01 9.44179770164805531873497557171504e-02 5.19337706403280308098402429095586e-02 3.68262322224391673228249821647751e-03
            8.53648252342703695028092170105083e-01 5.19337706409965516041182809203747e-02 9.44179770162997533677895489745424e-02 3.68262322222490025985797146290679e-03
            9.44179770156613751286300839637988e-02 8.53648252343519042817376885068370e-01 5.19337706408196098095686465967447e-02 3.68262322222055391018891157273174e-03
            9.44179770163206116828646941030456e-02 5.19337706404461307840847439365461e-02 8.53648252343233271410838369774865e-01 3.68262322224534224129888215770734e-03
            5.19337706404022908523998580676562e-02 8.53648252342816049598184235946974e-01 9.44179770167816734272037137998268e-02 3.68262322225092631616805149974425e-03
            5.19337706408202065544443826183851e-02 9.44179770164746412497436267585726e-02 8.53648252342705138318024182808585e-01 3.68262322222560932807877698280663e-03
            6.42663057493970235611868702108040e-01 3.34082493247154554971700690657599e-01 2.32544492588752094164306072343607e-02 3.80110733978111185044079789463467e-03
            6.42663057493039424628022970864549e-01 2.32544492590843546175882750048913e-02 3.34082493247876199937707042408874e-01 3.80110733981337293660440757037122e-03
            3.34082493247783773870906998126884e-01 6.42663057493066514069823824684136e-01 2.32544492591497675704204084468074e-02 3.80110733981773273038040628080125e-03
            3.34082493247255973845000198707567e-01 2.32544492588073331562625867263705e-02 6.42663057493936706876525022380520e-01 3.80110733978361549009750142147368e-03
            2.32544492588607973337921919210203e-02 6.42663057493850664592116572748637e-01 3.34082493247288558890772947052028e-01 3.80110733976986563814604025424160e-03
            2.32544492591022777805420673757908e-02 3.34082493247969403160624324300443e-01 6.42663057492928402325560455210507e-01 3.80110733982950152812230193433152e-03
            5.67421425923325672968644539650995e-01 4.09119706855485432139829526931862e-01 2.34588672211888948915259334171424e-02 4.08979954760176327849263699931726e-03
            5.67421425923091859999658481683582e-01 2.34588672212585509779803061292114e-02 4.09119706855649578614020356326364e-01 4.08979954755928944154508286601413e-03
            4.09119706855425480096499768478679e-01 5.67421425923261169010913818055997e-01 2.34588672213132953814351822074968e-02 4.08979954756468529891710872448130e-03
            4.09119706855585518745499484793982e-01 2.34588672211401734479796488130887e-02 5.67421425923274269642604394903174e-01 4.08979954758290596694703111779745e-03
            2.34588672211915282017624662103117e-02 5.67421425923561151272167535353219e-01 4.09119706855247344812198662111768e-01 4.08979954757752919153324100420832e-03
            2.34588672212738789946140371966976e-02 4.09119706855986309257389166305074e-01 5.67421425922739808278549844544614e-01 4.08979954757256961711542331272540e-03
            7.95883644812691470882271005393704e-01 1.49641910459035770708524637484516e-01 5.44744447282727584092043571217800e-02 4.61385528119594677531889459487502e-03
            7.95883644812847901306440689950250e-01 5.44744447286502411764708142527525e-02 1.49641910458501864455982399704226e-01 4.61385528118386442630871613346244e-03
            1.49641910457888716035057541375863e-01 7.95883644813761947922614581329981e-01 5.44744447283492805311766460363287e-02 4.61385528116337647469569205327389e-03
            1.49641910458972571262847850448452e-01 5.44744447285333069364021696401323e-02 7.95883644812494073228492652560817e-01 4.61385528123353302887288407418964e-03
            5.44744447284796137753737355069461e-02 7.95883644812258372880364731827285e-01 1.49641910459262006405367628758540e-01 4.61385528120587546513364785027989e-03
            5.44744447283740107490501713982667e-02 1.49641910458703203401498171842832e-01 7.95883644812922730338300425501075e-01 4.61385528115934064052883201156874e-03
            7.31688847721034707483056536148069e-01 2.12466086260515274464211188387708e-01 5.58450660184500180527322754642228e-02 5.20268756996140457199429008028346e-03
            7.31688847722387958327772139455192e-01 5.58450660178026955793306740361004e-02 2.12466086259809339154003282601479e-01 5.20268756993250407888451647409056e-03
            2.12466086259354897114448590400571e-01 7.31688847722972823817144671920687e-01 5.58450660176722513128311220498290e-02 5.20268756994749902861085999461466e-03
            2.12466086260680087072216792876134e-01 5.58450660185428257586970346437738e-02 7.31688847720777135741343499830691e-01 5.20268756998030698635027135878772e-03
            5.58450660185079370001481890994910e-02 7.31688847720892487913602053595241e-01 2.12466086260599595902931469026953e-01 5.20268756994296793089160857448405e-03
            5.58450660176888491470492681401083e-02 2.12466086259939790359396738494979e-01 7.31688847722371416004705224622739e-01 5.20268756992082938989119256234517e-03
            6.63321124311019305608283502806444e-01 2.80443055765804138967922654046561e-01 5.62358199231765554237938431469956e-02 5.61787388306443820351621099007389e-03
            6.63321124311019305608283502806444e-01 5.62358199233307098907630461326335e-02 2.80443055765649984500953451060923e-01 5.61787388315458224158360778233146e-03
            2.80443055765542181845262348360848e-01 6.63321124311019305608283502806444e-01 5.62358199234385125464541488327086e-02 5.61787388319449736140409612517033e-03
            2.80443055766081694724078943181667e-01 5.62358199230450495065269933547825e-02 6.63321124310873200258242832205724e-01 5.61787388305375230690419385837231e-03
            5.62358199231303007570303975626302e-02 6.63321124311254006755689260899089e-01 2.80443055765615678609492533723824e-01 5.61787388304078091211257728332384e-03
            5.62358199233420064100386071004323e-02 2.80443055765748405772086471188231e-01 6.63321124310909615573450537340250e-01 5.61787388315856950349314047343796e-03
            7.52105512554053334817183440463850e-01 1.48431512855705355358182373493037e-01 9.94629745902413098246341860431130e-02 6.10585546225158887906347970897514e-03
            7.52105512553738697612004671100294e-01 9.94629745906276119260525092613534e-02 1.48431512855633690461942819638352e-01 6.10585546229191946515646449711312e-03
            1.48431512856109171227814158555702e-01 7.52105512553738697612004671100294e-01 9.94629745901521866713324016018305e-02 6.10585546228902768112201115968674e-03
            1.48431512855414282636701273077051e-01 9.94629745906932954957468950851762e-02 7.52105512553892352478612792765489e-01 6.10585546226754920240420432264727e-03
            9.94629745905548645623639458790421e-02 7.52105512553738697612004671100294e-01 1.48431512855706437825631383020664e-01 6.10585546226101883587888963234036e-03
            9.94629745901721429302000387906446e-02 1.48431512855756869706524980756512e-01 7.52105512554070987363274980452843e-01 6.10585546228208965457984191971263e-03
            5.89842412473401966899189119430957e-01 3.53507535982312048883358102102648e-01 5.66500515442859842174527784663951e-02 6.10669701057133928329134775481180e-03
            5.89842412472958987912363681971328e-01 5.66500515446290500709558557446144e-02 3.53507535982411968955574366191286e-01 6.10669701056029690100501738925232e-03
            3.53507535982529985663092020331533e-01 5.89842412472691091096521631698124e-01 5.66500515447789787515375792281702e-02 6.10669701058331754889296760779871e-03
            3.53507535982596321488813373434823e-01 5.66500515441443891484496475641208e-02 5.89842412473259303240524786815513e-01 6.10669701055704516184929886435384e-03
            5.66500515442733276749720516818343e-02 5.89842412473763233471402145369211e-01 3.53507535981963383342474571691127e-01 6.10669701056666767297054221330654e-03
            5.66500515446690111609484574728413e-02 3.53507535982596321488813373434823e-01 5.89842412472734611839086937834509e-01 6.10669701057542282235379715871204e-03
            5.11476044318352696116392053227173e-01 4.31429040443182809649869113854947e-01 5.70949152384644942337388329178793e-02 6.45456202140728095562050015132627e-03
            5.11476044318822542500413419475080e-01 5.70949152385017075217454873836687e-02 4.31429040442675770794522804862936e-01 6.45456202136638138022739497046132e-03
            4.31429040442822320233773325526272e-01 5.11476044318545208788862055371283e-01 5.70949152386325264885158503602725e-02 6.45456202138116642841314529732699e-03
            4.31429040443462530340923422045307e-01 5.70949152383450203585013582596730e-02 5.11476044318192490933938643138390e-01 6.45456202139252886718079338379539e-03
            5.70949152384480629329743806010811e-02 5.11476044318667555366175747622037e-01 4.31429040442884326189698640519055e-01 6.45456202141345223438628764256464e-03
            5.70949152385571492840377061384061e-02 4.31429040443013944727823627545149e-01 5.11476044318428968438183801481500e-01 6.45456202138783644017827612060501e-03
            6.85135741196404968178512717713602e-01 2.13046127940230894948214768191974e-01 1.01818130863364136873272514094424e-01 7.20439116713822609583406375577397e-03
            6.85135741196668979213768579938915e-01 1.01818130862661060387353018086287e-01 2.13046127940669960398878401974798e-01 7.20439116719276666928051255922583e-03
            2.13046127941300844632621647178894e-01 6.85135741196165715116705996479141e-01 1.01818130862533440250672356341965e-01 7.20439116720144549083082452511917e-03
            2.13046127939914037296986748515337e-01 1.01818130863432915189648042542103e-01 6.85135741196652992002213977684733e-01 7.20439116713657637380840981222718e-03
            1.01818130863392544704915110287402e-01 6.85135741196353564852472572965780e-01 2.13046127940253904320400124561274e-01 7.20439116713638208477910040983261e-03
            1.01818130862524822144443703564320e-01 2.13046127940857032978527740851860e-01 6.85135741196618130999240747769363e-01 7.20439116719568100472015359514444e-03
            6.11188722458989031061094010510715e-01 2.86145793841027562542933537770296e-01 1.02665483699983406395972451718990e-01 8.06068774085997447054197095894779e-03
            6.11188722458106847845726861123694e-01 1.02665483700396562016798895911052e-01 2.86145793841496576259686435150797e-01 8.06068774085536184081934862888374e-03
            2.86145793842079221303009717303212e-01 6.11188722457293609480188933957834e-01 1.02665483700627113705650117481127e-01 8.06068774086064407380369800648623e-03
            2.86145793840845930056104862160282e-01 1.02665483699732676403648667928792e-01 6.11188722459421351906883046467556e-01 8.06068774086350983698601169180620e-03
            1.02665483699930296102031945792987e-01 6.11188722458957167660287268518005e-01 2.86145793841112494604317362245638e-01 8.06068774086888054086763588657050e-03
            1.02665483700433671221396991768415e-01 2.86145793841670770252250122211990e-01 6.11188722457895572404140693834051e-01 8.06068774085275802088190744143503e-03
            5.31047050461047831504401983693242e-01 3.65311627705897923323874465495464e-01 1.03641321833054245171723550811294e-01 8.65648115908654834649293974280226e-03
            5.31047050460632052981679862568853e-01 1.03641321833449345790612028395117e-01 3.65311627705918628983283724664943e-01 8.65648115905318094043252585834125e-03
            3.65311627706307484597658685743227e-01 5.31047050459976799352546095178695e-01 1.03641321833715660538643987820251e-01 8.65648115904955883781468628512812e-03
            3.65311627705974806268329757585889e-01 1.03641321832808885883281391215860e-01 5.31047050461216363359540082456078e-01 8.65648115909738689877084283352815e-03
            1.03641321833035579547122040366958e-01 5.31047050460862424259289582550991e-01 3.65311627706102037826951800525421e-01 8.65648115909780843657550519765209e-03
            1.03641321833532973339941918311524e-01 3.65311627705974806268329757585889e-01 5.31047050460492164880577092844760e-01 8.65648115904510753737532979812386e-03
            6.16755405363580533339984413032653e-01 2.23611895416044620388262842425320e-01 1.59632699220374846271752744542027e-01 8.89974956583711521163060353956098e-03
            6.16755405364591280381603155547054e-01 1.59632699220018936525633534984081e-01 2.23611895415389783092763309468864e-01 8.89974956582223822310062644191930e-03
            2.23611895415503914019694775561220e-01 6.16755405364456166239506273996085e-01 1.59632699220039864229647719184868e-01 8.89974956582909038083073482994223e-03
            2.23611895415582712098867546046677e-01 1.59632699220264823170012391528871e-01 6.16755405364152520242271293682279e-01 8.89974956582210638411645220458013e-03
            1.59632699220314533405939982912969e-01 6.16755405364404762913466129248263e-01 2.23611895415280703680593887838768e-01 8.89974956583001151899647851450936e-03
            1.59632699219956236680317829268461e-01 2.23611895415993855440461857142509e-01 6.16755405364049824612493466702290e-01 8.89974956582486459444325532786024e-03
            5.40939115423062610865656552050496e-01 2.97770116249244931605488773129764e-01 1.61290768327692457528854674819740e-01 9.88439332084446037329783507630054e-03
            5.40939115423441418961658655462088e-01 1.61290768328309741530546261856216e-01 2.97770116248248839507795082681696e-01 9.88439332082222989195319229338565e-03
            2.97770116248247340706711838720366e-01 5.40939115423052063746922613063361e-01 1.61290768328700595546365548216272e-01 9.88439332082658751732484603280682e-03
            2.97770116248895821975395392655628e-01 1.61290768327288530636920427241421e-01 5.40939115423815675143259795731865e-01 9.88439332084590019378289582618891e-03
            1.61290768327623290634420527567272e-01 5.40939115423788585701458941912279e-01 2.97770116248588179175271761778276e-01 9.88439332086586686099138887584559e-03
            1.61290768328408801179918441448535e-01 2.97770116248729010965945462885429e-01 5.40939115422862215609711711294949e-01 9.88439332080696259064112041414774e-03
            4.59837236520724612631738636991940e-01 3.77459830225164549855065843075863e-01 1.62702933254110837513195519932196e-01 1.04835098306981216315092453328361e-02
            4.59837236521587033877267458592542e-01 1.62702933254249698658000511386490e-01 3.77459830224163295220307645649882e-01 1.04835098306866707218443224292059e-02
            3.77459830223968673124090855708346e-01 4.59837236521400793964886588582885e-01 1.62702933254630532911022555708769e-01 1.04835098306785817062758425777247e-02
            3.77459830225057080266282127922750e-01 1.62702933253738080132677623623749e-01 4.59837236521204784089889017195674e-01 1.04835098307099975484257825542045e-02
            1.62702933254021270270683885428298e-01 4.59837236521242032072365191197605e-01 3.77459830224736669901375307745184e-01 1.04835098307131391326407765518525e-02
            1.62702933254432968723790509102400e-01 3.77459830224411874155521218199283e-01 4.59837236521155157120688272698317e-01 1.04835098306746993951366064834474e-02
            4.66649539183050188206891562003875e-01 3.03210043229955872323699850312551e-01 2.30140417586993939469408587683574e-01 1.15470205304319343403429698469154e-02
            4.66649539182556305494387061116868e-01 2.30140417587519352515812443016330e-01 3.03210043229924397500951727124630e-01 1.15470205304139417884501384037321e-02
            3.03210043229811321285893654930987e-01 4.66649539182124373226656643964816e-01 2.30140417588064360998600932362024e-01 1.15470205304243900279459467128618e-02
            3.03210043230049408613524519751081e-01 2.30140417586447293407658776231983e-01 4.66649539183503270223241088388022e-01 1.15470205304285707115230508179593e-02
            2.30140417586897821911051664756087e-01 4.66649539182707573381492238695500e-01 3.03210043230394576951880480919499e-01 1.15470205304487819747416565974163e-02
            2.30140417587695655932122917874949e-01 3.03210043229433678924067407933762e-01 4.66649539182870665143809674191289e-01 1.15470205304045239746990603180166e-02
            ];
        
        
    otherwise
        xyw=[];
end




pointset_stats=[
    1    1 1.92e-16    0    0
    2    3 1.42e-16    0    0
    4    6 1.53e-16    0    0
    5   10 3.05e-16    0    0
    7   15 3.61e-16    0    0
    8   21 5.62e-16    0    0
    10   28 4.72e-16    0    0
    12   36 4.44e-16    0    0
    14   45 5.00e-16    0    0
    15   55 4.44e-16    0    0
    17   66 5.34e-16    0    0
    19   78 8.53e-16    0    0
    20   91 4.30e-16    0    0
    22  105 5.69e-16    0    0
    24  120 6.18e-16    0    0
    25  136 6.38e-16    0    0
    27  153 5.00e-16    0    0
    28  171 5.27e-16    0    0
    30  190 6.42e-16    0    0
    32  210 6.83e-16    0    0
    ];





