
function [xyw,pointset_stats]=set_vioreanu_rokhlin_C_standard(deg)

%--------------------------------------------------------------------
% CUBATURE RULE ON THE SIMPLEX [0 0; 0 1; 1 0] WITH ADE EQUAL TO deg.
% THE CUBATURE RULE IS DESCRIBED IN STANDARD FORM.
%--------------------------------------------------------------------
% INPUT:
% deg: algebraic degree of precision (ADE) of the rule.
%--------------------------------------------------------------------
% OUTPUT:
% xyw: MATRIX OF NODES xyw(:,1:2) AND WEIGHTS xyw(:,3).
%      REMARK: The sum of the weights makes 1/2.
%
% pointset_stats:
%    statistics of the pointset stored in a matrix. Each rows contains
%    degree, cardinality of the full pointset, moment error w.r.t. orthn.
%    Dubiner basis,number of points with negative weights, number of
%    points outside the domain.
%--------------------------------------------------------------------
% SOURCE:
%  B. Vioreanu and V. Rokhlin,
%  "Spectra of Multiplication Operators as a Numerical Tool",
%  Methods and Algorithms for Scientific Computing,
%  SIAM J. Sci. Comput., 36,1 (2014), pp.267-288.
%--------------------------------------------------------------------
% | DEG | CARD | W- | PT.O|   M.E.   |Q.|
%--------------------------------------------------------------------
% |   1 |   1 |   0 |   0 | 1.92e-16 |PI|
% |   2 |   3 |   0 |   0 | 1.42e-16 |PI|
% |   4 |   6 |   0 |   0 | 1.53e-16 |PI|
% |   5 |  10 |   0 |   0 | 3.05e-16 |PI|
% |   7 |  15 |   0 |   0 | 3.61e-16 |PI|
% |   8 |  21 |   0 |   0 | 5.62e-16 |PI|
% |  10 |  28 |   0 |   0 | 4.72e-16 |PI|
% |  12 |  36 |   0 |   0 | 4.44e-16 |PI|
% |  14 |  45 |   0 |   0 | 5.00e-16 |PI|
% |  15 |  55 |   0 |   0 | 4.44e-16 |PI|
% |  17 |  66 |   0 |   0 | 5.34e-16 |PI|
% |  19 |  78 |   0 |   0 | 8.53e-16 |PI|
% |  20 |  91 |   0 |   0 | 4.30e-16 |PI|
% |  22 | 105 |   0 |   0 | 5.69e-16 |PI|
% |  24 | 120 |   0 |   0 | 6.18e-16 |PI|
% |  25 | 136 |   0 |   0 | 6.38e-16 |PI|
% |  27 | 153 |   0 |   0 | 5.00e-16 |PI|
% |  28 | 171 |   0 |   0 | 5.27e-16 |PI|
% |  30 | 190 |   0 |   0 | 6.42e-16 |PI|
% |  32 | 210 |   0 |   0 | 6.83e-16 |PI|
%--------------------------------------------------------------------
% LEGEND:
%--------------------------------------------------------------------
% DEG  : algebraic degree of precision
% CARD : cardinality of the pointset
% W-   : negative weights
% PT.O : points outside the simplex [0 0; 1 0; 0 1].
% M.E. : maximum moment error in inf norm.
% Q.: quality of the rule.
%        PI: pos. weights, points in simplex
%        NI: some neg. weights, points in simplex
%        P0: pos. weights, some points out of the simplex
%        N0: some neg. weights, some points out of the simplex
%--------------------------------------------------------------------



switch deg
    
    case   1
        
        % ALG. DEG.:    1
        % PTS CARD.:    1
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.9e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 5.00000000000000000000000000000000e-01
            ];
        
    case   2
        
        % ALG. DEG.:    2
        % PTS CARD.:    3
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.4e-16
        
        xyw=[
            1.66666666666666601903656896865868e-01 1.66666666666666601903656896865868e-01 1.66666666666666629659232512494782e-01
            1.66666666666666796192686206268263e-01 6.66666666666666518636930049979128e-01 1.66666666666666768437110590639350e-01
            6.66666666666666740681534975010436e-01 1.66666666666666490881354434350214e-01 1.66666666666666657414808128123695e-01
            ];
        
    case   4
        
        % ALG. DEG.:    4
        % PTS CARD.:    6
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.5e-16
        
        xyw=[
            9.15762135097707430375635340169538e-02 9.15762135097707430375635340169538e-02 5.49758718276609284481004635836143e-02
            9.15762135097707430375635340169538e-02 8.16847572980458513924872931966092e-01 5.49758718276609284481004635836143e-02
            8.16847572980458513924872931966092e-01 9.15762135097707430375635340169538e-02 5.49758718276609284481004635836143e-02
            4.45948490915964890213274429697776e-01 4.45948490915964890213274429697776e-01 1.11690794839005722027813760632853e-01
            4.45948490915964890213274429697776e-01 1.08103018168070219573451140604448e-01 1.11690794839005722027813760632853e-01
            1.08103018168070219573451140604448e-01 4.45948490915964890213274429697776e-01 1.11690794839005722027813760632853e-01
            ];
        
    case   5
        
        % ALG. DEG.:    5
        % PTS CARD.:   10
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.1e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 9.94826852298182351397315414942568e-02
            7.11129945334409846458001425162365e-02 7.11129945334410956681026050318906e-02 3.01394902035703009246514483265855e-02
            7.11129945334409985235879503306933e-02 8.57774010933118002952824099338613e-01 3.01394902035702801079697366049004e-02
            8.57774010933118002952824099338613e-01 7.11129945334409291346489112584095e-02 3.01394902035702627607349768368294e-02
            6.14926166643125493571631068334682e-01 3.13979397676981020204323158395709e-01 5.16831406932451511981874148204952e-02
            6.14926166643125493571631068334682e-01 7.10944356798936805130750826720032e-02 5.16831406932451997704447421710938e-02
            3.13979397676980853670869464622228e-01 6.14926166643125493571631068334682e-01 5.16831406932451997704447421710938e-02
            3.13979397676981020204323158395709e-01 7.10944356798934445906823498262384e-02 5.16831406932451303815057030988100e-02
            7.10944356798934307128945420117816e-02 6.14926166643125493571631068334682e-01 5.16831406932451026259300874698965e-02
            7.10944356798936805130750826720032e-02 3.13979397676981020204323158395709e-01 5.16831406932451650759752226349519e-02
            ];
        
    case   7
        
        % ALG. DEG.:    7
        % PTS CARD.:   15
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.6e-16
        
        xyw=[
            5.12530798414518040662102293936186e-02 5.12530798414517624328468059502484e-02 1.60066673649398305556701416207943e-02
            5.12530798414517624328468059502484e-02 8.97493840317096447378730772470590e-01 1.60066673649398097389884298991092e-02
            8.97493840317096447378730772470590e-01 5.12530798414518040662102293936186e-02 1.60066673649398236167762377135659e-02
            4.74110794051135264748353392860736e-01 4.74110794051135320259504624118563e-01 3.24399187735258873188648465202277e-02
            4.74110794051135209237202161602909e-01 5.17784118977294149921419830207014e-02 3.24399187735258803799709426129994e-02
            5.17784118977294566255054064640717e-02 4.74110794051135320259504624118563e-01 3.24399187735258873188648465202277e-02
            2.42184756344485996315540887735551e-01 2.42184756344485996315540887735551e-01 6.35044628471655714374577428316115e-02
            2.42184756344486107337843350251205e-01 5.15630487311027785324313299497589e-01 6.35044628471655991930333584605251e-02
            5.15630487311027785324313299497589e-01 2.42184756344486024071116503364465e-01 6.35044628471656130708211662749818e-02
            7.18483300897606480539536732976558e-01 2.33346892984868881093518666602904e-01 2.73578088405176979291955063899877e-02
            7.18483300897606480539536732976558e-01 4.81698061175246383669446004205383e-02 2.73578088405176840514076985755310e-02
            2.33346892984868853337943050973990e-01 7.18483300897606480539536732976558e-01 2.73578088405176875208546505291451e-02
            2.33346892984868908849094282231817e-01 4.81698061175245759168994652554829e-02 2.73578088405176528263851309930033e-02
            4.81698061175246730614141199566802e-02 7.18483300897606480539536732976558e-01 2.73578088405176875208546505291451e-02
            4.81698061175246106113689847916248e-02 2.33346892984868908849094282231817e-01 2.73578088405176667041729388074600e-02
            ];
        
    case   8
        
        % ALG. DEG.:    8
        % PTS CARD.:   21
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.6e-16
        
        xyw=[
            3.14057488009515869964438650185912e-02 3.14057488009516008742316728330479e-02 6.85725362752989762299415943402892e-03
            3.14057488009515453630804415752209e-02 9.37188502398096923151626924664015e-01 6.85725362752988201048287564276507e-03
            9.37188502398096923151626924664015e-01 3.14057488009515384241865376679925e-02 6.85725362752988547992982759637925e-03
            1.96019357807156596740227882946783e-01 1.96019357807156568984652267317870e-01 3.71988045099158892714541480017942e-02
            1.96019357807156707762530345462437e-01 6.07961284385686751008393002848607e-01 3.71988045099159170270297636307077e-02
            6.07961284385686751008393002848607e-01 1.96019357807156596740227882946783e-01 3.71988045099158962103480519090226e-02
            4.04461999739164801415114425253705e-01 4.04461999739164801415114425253705e-01 4.25072661072600777032448604586534e-02
            4.04461999739164801415114425253705e-01 1.91076000521670452680922380750417e-01 4.25072661072600985199265721803386e-02
            1.91076000521670313903044302605849e-01 4.04461999739164801415114425253705e-01 4.25072661072600915810326682731102e-02
            7.87642410206076615253323325305246e-01 1.71130424662843866334682729757333e-01 1.92029225190513477827813915155275e-02
            7.87642410206076615253323325305246e-01 4.12271651310795253508878488446499e-02 1.92029225190513616605691993299843e-02
            1.71130424662843894090258345386246e-01 7.87642410206076615253323325305246e-01 1.92029225190513477827813915155275e-02
            1.71130424662843894090258345386246e-01 4.12271651310794767786305214940512e-02 1.92029225190513443133344395619133e-02
            4.12271651310795322897817527518782e-02 7.87642410206076615253323325305246e-01 1.92029225190513616605691993299843e-02
            4.12271651310795808620390801024769e-02 1.71130424662843894090258345386246e-01 1.92029225190513512522283434691417e-02
            5.84632756543619902522834763658466e-01 3.76058771577794492824153849142021e-01 2.08487486919290317122221978252128e-02
            5.84632756543619902522834763658466e-01 3.93084718785856948586321379934816e-02 2.08487486919290178344343900107560e-02
            3.76058771577794381801851386626367e-01 5.84632756543619902522834763658466e-01 2.08487486919290004871996302426851e-02
            3.76058771577794437313002617884194e-01 3.93084718785857503697833692513086e-02 2.08487486919290213038813419643702e-02
            3.93084718785857087364199458079383e-02 5.84632756543619902522834763658466e-01 2.08487486919290317122221978252128e-02
            3.93084718785856115919052911067411e-02 3.76058771577794381801851386626367e-01 2.08487486919289900788587743818425e-02
            ];
        
    case  10
        
        % ALG. DEG.:   10
        % PTS CARD.:   28
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.7e-16
        
        xyw=[
            3.33333333333333536874221181278699e-01 3.33333333333333314829616256247391e-01 4.04081300254218525602745160085760e-02
            2.27066403128517654841367345852632e-02 2.27066403128517411980080709099639e-02 3.69081644809862545464351768487177e-03
            2.27066403128517481369019748171922e-02 9.54586719374296510665089954272844e-01 3.69081644809862805672873165008241e-03
            9.54586719374296621687392416788498e-01 2.27066403128517065035385513738220e-02 3.69081644809861721470700679503807e-03
            4.85348136374768979184324280140572e-01 4.85348136374769034695475511398399e-01 1.32394430007485800621092053575012e-02
            4.85348136374768979184324280140572e-01 2.93037272504619444868367850176583e-02 1.32394430007485679190448735198515e-02
            2.93037272504620277535636319043988e-02 4.85348136374768979184324280140572e-01 1.32394430007485887357265852415367e-02
            1.55015517336222874433815377415158e-01 1.55015517336222957700542224301898e-01 2.50546571154375528889257651599110e-02
            1.55015517336222985456117839930812e-01 6.89968965327554029087764320138376e-01 2.50546571154375632972666210207535e-02
            6.89968965327554029087764320138376e-01 1.55015517336222985456117839930812e-01 2.50546571154375598278196690671393e-02
            8.38551707380394306667881210159976e-01 1.30053450518161112237081056264287e-01 1.17510521064424847126339557235042e-02
            8.38551707380394306667881210159976e-01 3.13948421014445602783560218540515e-02 1.17510521064424881820809076771184e-02
            1.30053450518161306526110365666682e-01 8.38551707380394084623276285128668e-01 1.17510521064424864473574317003113e-02
            1.30053450518161251014959134408855e-01 3.13948421014445255838865023179096e-02 1.17510521064424812431870037698900e-02
            3.13948421014445602783560218540515e-02 8.38551707380394195645578747644322e-01 1.17510521064424933862513356075397e-02
            3.13948421014445255838865023179096e-02 1.30053450518161195503807903151028e-01 1.17510521064424725695696238858545e-02
            6.66920601059608042504578406806104e-01 3.02790097367536925609954323590500e-01 1.42630613728036703680146501937998e-02
            6.66920601059607931482275944290450e-01 3.02893015728550631104898371859235e-02 1.42630613728036755721850781242210e-02
            3.02790097367536981121105554848327e-01 6.66920601059607931482275944290450e-01 1.42630613728036651638442222633785e-02
            3.02790097367536981121105554848327e-01 3.02893015728550943355124047684512e-02 1.42630613728036773069085541010281e-02
            3.02893015728550631104898371859235e-02 6.66920601059607931482275944290450e-01 1.42630613728036790416320300778352e-02
            3.02893015728551116827471645365222e-02 3.02790097367536925609954323590500e-01 1.42630613728036755721850781242210e-02
            5.20560306058803101691978554299567e-01 3.30714245940923079469797585261404e-01 2.95920732343744712078414949019134e-02
            5.20560306058802990669676091783913e-01 1.48725448000273791082648244810116e-01 2.95920732343744885550762546699843e-02
            3.30714245940923190492100047777058e-01 5.20560306058802990669676091783913e-01 2.95920732343744816161823507627560e-02
            3.30714245940923079469797585261404e-01 1.48725448000273846593799476067943e-01 2.95920732343744885550762546699843e-02
            1.48725448000273874349375091696857e-01 5.20560306058802990669676091783913e-01 2.95920732343744885550762546699843e-02
            1.48725448000273929860526322954684e-01 3.30714245940923079469797585261404e-01 2.95920732343745093717579663916695e-02
            ];
        
    case  12
        
        % ALG. DEG.:   12
        % PTS CARD.:   36
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.4e-16
        
        xyw=[
            1.93652109024277166438032793394086e-02 1.93652109024277131743563273857944e-02 2.57067510984239876170587280057589e-03
            1.93652109024277270521441352002512e-02 9.61269578195144580590181249135640e-01 2.57067510984239138913109989914574e-03
            9.61269578195144580590181249135640e-01 1.93652109024277339910380391074796e-02 2.57067510984239355753544487015461e-03
            1.20430831569190488483656054086168e-01 1.20430831569190682772685363488563e-01 1.58198213142406486630964224104900e-02
            1.20430831569190530117019477529539e-01 7.59138336861618912010385429312009e-01 1.58198213142406209075208067815765e-02
            7.59138336861618912010385429312009e-01 1.20430831569190530117019477529539e-01 1.58198213142406104991799509207340e-02
            4.40005690210435673748179397080094e-01 4.40005690210435562725876934564440e-01 2.33708295855031729371109605608581e-02
            4.40005690210435562725876934564440e-01 1.19988619579128805159307091798837e-01 2.33708295855031798760048644680865e-02
            1.19988619579128763525943668355467e-01 4.40005690210435840281633090853575e-01 2.33708295855031521204292488391729e-02
            2.71524632000497700978769444191130e-01 2.71524632000497700978769444191130e-01 3.12289127691522533369905545441725e-02
            2.71524632000497700978769444191130e-01 4.56950735999004598042461111617740e-01 3.12289127691522637453314104050150e-02
            4.56950735999004653553612342875567e-01 2.71524632000497645467618212933303e-01 3.12289127691522533369905545441725e-02
            8.69373866882279466850036442338023e-01 1.06407946399805108939062847639434e-01 7.40717691450160174021499770447008e-03
            8.69373866882279466850036442338023e-01 2.42181867179154276803476619761568e-02 7.40717691450160347493847368127717e-03
            1.06407946399805122816850655453891e-01 8.69373866882279466850036442338023e-01 7.40717691450160000549152172766298e-03
            1.06407946399805136694638463268348e-01 2.42181867179154103331129022080859e-02 7.40717691450160694438542563489136e-03
            2.42181867179154033942189983008575e-02 8.69373866882279577872338904853677e-01 7.40717691450159480132109379724170e-03
            2.42181867179154207414537580689284e-02 1.06407946399805108939062847639434e-01 7.40717691450159393395935580883815e-03
            5.62233442343925027273598971078172e-01 4.14262053470467184190795251197414e-01 9.15466123399987299902313964139466e-03
            5.62233442343925027273598971078172e-01 2.35045041856077469022423542810429e-02 9.15466123399986952957618768778048e-03
            4.14262053470467184190795251197414e-01 5.62233442343925027273598971078172e-01 9.15466123399986779485271171097338e-03
            4.14262053470467184190795251197414e-01 2.35045041856077538411362581882713e-02 9.15466123399986259068228378055210e-03
            2.35045041856077399633484503738146e-02 5.62233442343925027273598971078172e-01 9.15466123399987299902313964139466e-03
            2.35045041856077295550075945129720e-02 4.14262053470467184190795251197414e-01 9.15466123399985912123533182693791e-03
            7.25103351867971257505018911615480e-01 2.52014086602536824255338387956726e-01 9.51292473557083066715289731973826e-03
            7.25103351867971257505018911615480e-01 2.28825615294920119147104031753770e-02 9.51292473557083066715289731973826e-03
            2.52014086602536824255338387956726e-01 7.25103351867971146482716449099826e-01 9.51292473557082719770594536612407e-03
            2.52014086602536824255338387956726e-01 2.28825615294920257924982109898338e-02 9.51292473557084801438765708780920e-03
            2.28825615294919980369225953609202e-02 7.25103351867971257505018911615480e-01 9.51292473557084454494070513419501e-03
            2.28825615294919876285817395000777e-02 2.52014086602536824255338387956726e-01 9.51292473557083587132332525015954e-03
            6.17412891519986017918597553943982e-01 2.66109864468771051271289707074175e-01 2.07634510598917881862846002150036e-02
            6.17412891519986017918597553943982e-01 1.16477244011242916932324931167386e-01 2.07634510598917881862846002150036e-02
            2.66109864468771106782440938332002e-01 6.17412891519985906896295091428328e-01 2.07634510598917951251785041222320e-02
            2.66109864468771106782440938332002e-01 1.16477244011242972443476162425213e-01 2.07634510598917812473906963077752e-02
            1.16477244011242972443476162425213e-01 6.17412891519986017918597553943982e-01 2.07634510598917743084967924005468e-02
            1.16477244011242930810112738981843e-01 2.66109864468771384338197094621137e-01 2.07634510598917916557315521686178e-02
            ];
        
    case  14
        
        % ALG. DEG.:   14
        % PTS CARD.:   45
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.0e-16
        
        xyw=[
            1.88250991695916025658874559667311e-02 1.88250991695915990964405040131169e-02 2.31145558557661586276954501784076e-03
            1.88250991695915852186526961986601e-02 9.62349801660816739357073856808711e-01 2.31145558557661499540780702943721e-03
            9.62349801660816739357073856808711e-01 1.88250991695915990964405040131169e-02 2.31145558557661673013128300624430e-03
            4.90249109938372285011354279049556e-01 4.90249109938372229500203047791729e-01 7.17144341823516221628986855307630e-03
            4.90249109938372285011354279049556e-01 1.95017801232554889578896251123297e-02 7.17144341823516828782203447190113e-03
            1.95017801232555028356774329267864e-02 4.90249109938372285011354279049556e-01 7.17144341823517696143941435593661e-03
            8.19725416516456917292998696211725e-02 8.19725416516453586623924820742104e-02 8.69911146973070120713700248415989e-03
            8.19725416516457056070876774356293e-02 8.36054916696708616541400260757655e-01 8.69911146973072202381871420584503e-03
            8.36054916696708616541400260757655e-01 8.19725416516456084625730227344320e-02 8.69911146973072028909523822903793e-03
            2.20530835565724586677660568057036e-01 2.20530835565724503410933721170295e-01 2.26930438835609080139299464917713e-02
            2.20530835565724447899782489912468e-01 5.58938328868550771133527632628102e-01 2.26930438835609080139299464917713e-02
            5.58938328868550549088922707596794e-01 2.20530835565724753211114261830517e-01 2.26930438835609149528238503989996e-02
            3.86124189278291174431956278567668e-01 3.86124189278291507498863666114630e-01 2.60961464380900509674177101260284e-02
            3.86124189278291285454258741083322e-01 2.27751621443417290313604439688788e-01 2.60961464380900509674177101260284e-02
            2.27751621443417318069180055317702e-01 3.86124189278291229943107509825495e-01 2.60961464380900509674177101260284e-02
            8.87822775364344174420239141909406e-01 9.67912764708031281291056302507059e-02 4.10204095024180835638016873190281e-03
            8.87822775364344174420239141909406e-01 1.53859481648527408187421272600659e-02 4.10204095024180488693321677828862e-03
            9.67912764708029893512275521061383e-02 8.87822775364344285442541604425060e-01 4.10204095024180748901843074349927e-03
            9.67912764708030587401665911784221e-02 1.53859481648526350006100926748331e-02 4.10204095024179014178367097542832e-03
            1.53859481648527599007003630049439e-02 8.87822775364344063397936679393752e-01 4.10204095024181442791233465072764e-03
            1.53859481648527425534656032368730e-02 9.67912764708030864957422068073356e-02 4.10204095024181009110364470870991e-03
            7.67286163241992502648258778208401e-01 2.14478157320871859470301501460199e-01 5.94679596697558158352459045659089e-03
            7.67286163241992502648258778208401e-01 1.82356794371357107398257113572981e-02 5.94679596697560066548282620146892e-03
            2.14478157320871803959150270202372e-01 7.67286163241992502648258778208401e-01 5.94679596697560066548282620146892e-03
            2.14478157320871803959150270202372e-01 1.82356794371356413508866722850144e-02 5.94679596697558245088632844499443e-03
            1.82356794371356274730988644705576e-02 7.67286163241992502648258778208401e-01 5.94679596697556510365156867692349e-03
            1.82356794371356482897805761922427e-02 2.14478157320871803959150270202372e-01 5.94679596697558331824806643339798e-03
            6.33349939282431662057604171423009e-01 3.48145146811778050111030324842432e-01 6.56292539161341427489393396399464e-03
            6.33349939282431551035301708907355e-01 1.85049139057902913008124556881739e-02 6.56292539161341861170262390601238e-03
            3.48145146811778105622181556100259e-01 6.33349939282431551035301708907355e-01 6.56292539161342468323478982483721e-03
            3.48145146811778105622181556100259e-01 1.85049139057902843619185517809456e-02 6.56292539161341687697914792920528e-03
            1.85049139057903155869411193634733e-02 6.33349939282431773079906633938663e-01 6.56292539161341687697914792920528e-03
            1.85049139057903294647289271779300e-02 3.48145146811777994599879093584605e-01 6.56292539161342034642609988281947e-03
            7.07553018466495098337531999277417e-01 2.00642015647370303144114700444334e-01 1.44656152357198423269446507788416e-02
            7.07553018466495098337531999277417e-01 9.18049658861347234184435706083605e-02 1.44656152357198284491568429643849e-02
            2.00642015647370219877387853557593e-01 7.07553018466495098337531999277417e-01 1.44656152357198249797098910107707e-02
            2.00642015647370303144114700444334e-01 9.18049658861345291294142612059659e-02 1.44656152357198197755394630803494e-02
            9.18049658861345013738386455770524e-02 7.07553018466495098337531999277417e-01 1.44656152357198353880507468716132e-02
            9.18049658861345985183533002782497e-02 2.00642015647369859054904850381718e-01 1.44656152357198232449864150339636e-02
            5.41383214828122127926235407358035e-01 3.61345725637031767174534024888999e-01 1.87703553911860061831351487171560e-02
            5.41383214828122349970840332389344e-01 9.72710595348459799991402974228549e-02 1.87703553911859992442412448099276e-02
            3.61345725637031489618777868599864e-01 5.41383214828122460993142794904998e-01 1.87703553911859957747942928563134e-02
            3.61345725637031711663382793631172e-01 9.72710595348459522435646817939414e-02 1.87703553911859888359003889490850e-02
            9.72710595348459661213524896083982e-02 5.41383214828122127926235407358035e-01 1.87703553911860061831351487171560e-02
            9.72710595348459938769281052373117e-02 3.61345725637031323085324174826383e-01 1.87703553911859923053473409026992e-02
            ];
        
    case  15
        
        % ALG. DEG.:   15
        % PTS CARD.:   55
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.4e-16
        
        xyw=[
            3.33333333333333370340767487505218e-01 3.33333333333333314829616256247391e-01 2.23876490962987353960933489815943e-02
            1.76519929574458882270970860872694e-02 1.76519929574459194521196536697971e-02 1.95605020301731170537240522833144e-03
            1.76519929574459229215666056234113e-02 9.64696014085108188851336308289319e-01 1.95605020301731864426630913555982e-03
            9.64696014085108188851336308289319e-01 1.76519929574459298604605095306397e-02 1.95605020301731604218109517034918e-03
            7.71703842193710437546982916501292e-02 7.71703842193710021213348682067590e-02 8.26686281858202316741301984848178e-03
            7.71703842193708772212445978766482e-02 8.45659231561258217801935188617790e-01 8.26686281858201449379563996444631e-03
            8.45659231561258106779632726102136e-01 7.71703842193709327323958291344752e-02 8.26686281858201622851911594125340e-03
            4.59371797617381893985566421179101e-01 4.59371797617381893985566421179101e-01 1.03082351778196849856339412099260e-02
            4.59371797617381893985566421179101e-01 8.12564047652361287621403107550577e-02 1.03082351778196502911644216737841e-02
            8.12564047652361703955037341984280e-02 4.59371797617381893985566421179101e-01 1.03082351778196606995052775346267e-02
            2.04922897588778768485084924577677e-01 2.04922897588778796240660540206591e-01 1.59826993649305748590450093615800e-02
            2.04922897588778823996236155835504e-01 5.90154204822442407518678919586819e-01 1.59826993649305817979389132688084e-02
            5.90154204822442407518678919586819e-01 2.04922897588778851751811771464418e-01 1.59826993649305783284919613151942e-02
            9.03429699782028916565934650861891e-01 8.35136443388275473420634398280527e-02 2.83696446768514753261625216396169e-03
            9.03429699782028805543632188346237e-01 1.30566558791435395614488612636706e-02 2.83696446768515447151015607119007e-03
            8.35136443388276306087902867147932e-02 9.03429699782028805543632188346237e-01 2.83696446768515056838233512337411e-03
            8.35136443388274779531244007557689e-02 1.30566558791435829295357606838479e-02 2.83696446768515837463797701900603e-03
            1.30566558791435135405967216115641e-02 9.03429699782028916565934650861891e-01 2.83696446768514753261625216396169e-03
            1.30566558791435829295357606838479e-02 8.35136443388275057087000163846824e-02 2.83696446768515794095710802480426e-03
            8.10386575943458420390186347503914e-01 1.70743649147708009827795194723876e-01 4.48953911575521082827355812128189e-03
            8.10386575943458420390186347503914e-01 1.88697749088335836598062655866670e-02 4.48953911575521169563529610968544e-03
            1.70743649147708009827795194723876e-01 8.10386575943458420390186347503914e-01 4.48953911575520388937965421405352e-03
            1.70743649147708009827795194723876e-01 1.88697749088335316181019862824542e-02 4.48953911575521776716746202851027e-03
            1.88697749088335732514654097258244e-02 8.10386575943458420390186347503914e-01 4.48953911575520475674139220245706e-03
            1.88697749088335385569958901896825e-02 1.70743649147708009827795194723876e-01 4.48953911575520475674139220245706e-03
            7.02894663916247330881503785349196e-01 2.80682296607787540310141594090965e-01 5.35829842769160354098989529347818e-03
            7.02894663916247330881503785349196e-01 1.64230394759651843195058518176666e-02 5.35829842769160701043684724709237e-03
            2.80682296607787429287839131575311e-01 7.02894663916247441903806247864850e-01 5.35829842769160440835163328188173e-03
            2.80682296607787651332444056606619e-01 1.64230394759652051361875635393517e-02 5.35829842769160614307510925868883e-03
            1.64230394759651877889528037712807e-02 7.02894663916247330881503785349196e-01 5.35829842769160874516032322389947e-03
            1.64230394759651981972936596321233e-02 2.80682296607787484798990362833138e-01 5.35829842769160440835163328188173e-03
            5.64636464683940353381785826059058e-01 4.19465101758232095097156388874282e-01 5.86179021515446756207534662053149e-03
            5.64636464683940353381785826059058e-01 1.58984335578275411127169292058170e-02 5.86179021515446669471360863212794e-03
            4.19465101758231984074853926358628e-01 5.64636464683940464404088288574712e-01 5.86179021515447016416056058574213e-03
            4.19465101758231984074853926358628e-01 1.58984335578275341738230252985886e-02 5.86179021515445715373449075968892e-03
            1.58984335578275480516108331130454e-02 5.64636464683940353381785826059058e-01 5.86179021515447103152229857414568e-03
            1.58984335578275133571413135769035e-02 4.19465101758231984074853926358628e-01 5.86179021515446062318144271330311e-03
            5.84478584036716264371591478266055e-01 3.33042061006165956982272291497793e-01 1.26121766509501454650576945937246e-02
            5.84478584036716264371591478266055e-01 8.24793549571176815016215755349549e-02 1.26121766509501506692281225241459e-02
            3.33042061006165901471121060239966e-01 5.84478584036716375393893940781709e-01 1.26121766509501454650576945937246e-02
            3.33042061006166123515725985271274e-01 8.24793549571177092571971911638684e-02 1.26121766509501385261637906864962e-02
            8.24793549571176953794093833494117e-02 5.84478584036716264371591478266055e-01 1.26121766509501489345046465473388e-02
            8.24793549571176953794093833494117e-02 3.33042061006166123515725985271274e-01 1.26121766509501471997811705705317e-02
            7.17030852299470722321927951270482e-01 1.92513233859061272390533758880338e-01 1.29099875631297419942278281723702e-02
            7.17030852299470611299625488754828e-01 9.04559138414681024320529445503780e-02 1.29099875631297489331217320795986e-02
            1.92513233859061161368231296364684e-01 7.17030852299470722321927951270482e-01 1.29099875631297454636747801259844e-02
            1.92513233859061355657260605767078e-01 9.04559138414680607986895211070077e-02 1.29099875631297506678452080564057e-02
            9.04559138414681024320529445503780e-02 7.17030852299470611299625488754828e-01 1.29099875631297524025686840332128e-02
            9.04559138414680746764773289214645e-02 1.92513233859061355657260605767078e-01 1.29099875631297541372921600100199e-02
            4.72245408246903863869192718993872e-01 3.36118950295532514793706013733754e-01 1.72763782614090734279788108551656e-02
            4.72245408246903919380343950251699e-01 1.91635641457563538070374420385633e-01 1.72763782614090838363196667160082e-02
            3.36118950295532514793706013733754e-01 4.72245408246903919380343950251699e-01 1.72763782614090664890849069479373e-02
            3.36118950295532514793706013733754e-01 1.91635641457563593581525651643460e-01 1.72763782614090838363196667160082e-02
            1.91635641457563593581525651643460e-01 4.72245408246903919380343950251699e-01 1.72763782614090768974257628087798e-02
            1.91635641457563621337101267272374e-01 3.36118950295532514793706013733754e-01 1.72763782614090803668727147623940e-02
            ];
        
    case  17
        
        % ALG. DEG.:   17
        % PTS CARD.:   66
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.3e-16
        
        xyw=[
            1.33804426737547284276264036861903e-02 1.33804426737547266929029277093832e-02 1.13328753458473516918525447749744e-03
            1.33804426737546659775812685211349e-02 9.73239114652490688861519174679415e-01 1.13328753458472931449352305577349e-03
            9.73239114652490577839216712163761e-01 1.33804426737547110803916439181194e-02 1.13328753458472953133395755287438e-03
            4.92937508753781006287653099207091e-01 4.92937508753780784243048174175783e-01 4.60230524156527830842433601787889e-03
            4.92937508753781006287653099207091e-01 1.41249824924381661012118271969484e-02 4.60230524156527570633912205266824e-03
            1.41249824924381556928709713361059e-02 4.92937508753781006287653099207091e-01 4.60230524156527570633912205266824e-03
            6.86030848984772734233672508707969e-02 6.86030848984772734233672508707969e-02 6.02728414491514492834500771323292e-03
            6.86030848984773844456697133864509e-02 8.62793830203045342130963035742752e-01 6.02728414491514579570674570163646e-03
            8.62793830203045342130963035742752e-01 6.86030848984773150567306743141671e-02 6.02728414491515273460064960886484e-03
            1.63281281675213801873880470338918e-01 1.63281281675213940651758548483485e-01 1.11721445806285679669400678903912e-02
            1.63281281675213718607153623452177e-01 6.73437436649572340741087828064337e-01 1.11721445806285766405574477744267e-02
            6.73437436649572340741087828064337e-01 1.63281281675213718607153623452177e-01 1.11721445806285679669400678903912e-02
            4.17510360602973917831803873923491e-01 4.17510360602973806809501411407837e-01 1.56328347760772250196747990003132e-02
            4.17510360602973695787198948892183e-01 1.64979278794052386380997177184327e-01 1.56328347760772146113339431394706e-02
            1.64979278794052275358694714668673e-01 4.17510360602973917831803873923491e-01 1.56328347760772146113339431394706e-02
            2.86487437482899232055189031598275e-01 2.86487437482899343077491494113929e-01 1.84063259538869879172739985051521e-02
            2.86487437482899343077491494113929e-01 4.27025125034201313845017011772143e-01 1.84063259538869948561679024123805e-02
            4.27025125034201369356168243029970e-01 2.86487437482899232055189031598275e-01 1.84063259538869913867209504587663e-02
            9.22032114610694231160437084326986e-01 6.50938297812631572192287876532646e-02 2.22858734349179986311551004973808e-03
            9.22032114610694453205042009358294e-01 1.28740556080425300883307571098157e-02 2.22858734349180246520072401494872e-03
            6.50938297812631294636531720243511e-02 9.22032114610694342182739546842640e-01 2.22858734349180376624333099755404e-03
            6.50938297812632682415312501689186e-02 1.28740556080424867202438576896384e-02 2.22858734349180463360506898595759e-03
            1.28740556080425474355655168778867e-02 9.22032114610694231160437084326986e-01 2.22858734349180680200941395696645e-03
            1.28740556080424745771795258519887e-02 6.50938297812632404859556345400051e-02 2.22858734349179856207290306713276e-03
            8.42560091151481893589902938401792e-01 1.43419095062200824264309062527900e-01 3.15561337275237175845066239787684e-03
            8.42560091151481893589902938401792e-01 1.40208137863172769416175711398864e-02 3.15561337275237522789761435149103e-03
            1.43419095062200907531035909414641e-01 8.42560091151481782567600475886138e-01 3.15561337275237566157848334569280e-03
            1.43419095062200907531035909414641e-01 1.40208137863173289833218504440993e-02 3.15561337275237522789761435149103e-03
            1.40208137863172873499584270007290e-02 8.42560091151481893589902938401792e-01 3.15561337275236438587588949644669e-03
            1.40208137863173220444279465368709e-02 1.43419095062200907531035909414641e-01 3.15561337275237262581240038628039e-03
            7.43449702189565297949513933417620e-01 2.43155281870591571946604858567298e-01 3.82284387196494494093323979200250e-03
            7.43449702189565297949513933417620e-01 1.33950159398430919399647365253259e-02 3.82284387196494797669932275141491e-03
            2.43155281870591710724482936711865e-01 7.43449702189565186927211470901966e-01 3.82284387196494971142279872822201e-03
            2.43155281870591710724482936711865e-01 1.33950159398430850010708326180975e-02 3.82284387196494624197584677460782e-03
            1.33950159398431006135821164093613e-02 7.43449702189565297949513933417620e-01 3.82284387196495187982714369923087e-03
            1.33950159398430676538360728500265e-02 2.43155281870591710724482936711865e-01 3.82284387196494754301845375721314e-03
            6.22112357616232514168075340421638e-01 3.65088603079598228440261209470918e-01 4.20555642622647083556097769019289e-03
            6.22112357616232292123470415390329e-01 1.27990393041693493320076768782201e-02 4.20555642622646910083750171338579e-03
            3.65088603079598394973714903244399e-01 6.22112357616232292123470415390329e-01 4.20555642622647430500792964380707e-03
            3.65088603079598450484866134502226e-01 1.27990393041693961695415282520116e-02 4.20555642622648124390183355103545e-03
            1.27990393041693597403485327390626e-02 6.22112357616232403145772877905983e-01 4.20555642622647950917835757422836e-03
            1.27990393041693614750720087158697e-02 3.65088603079598394973714903244399e-01 4.20555642622647430500792964380707e-03
            5.17329851757723613303596721380018e-01 4.12641979622127230697969935135916e-01 8.41713097501839077607677097603300e-03
            5.17329851757723613303596721380018e-01 7.00281686201491559984333434840664e-02 8.41713097501838904135329499922591e-03
            4.12641979622127286209121166393743e-01 5.17329851757723613303596721380018e-01 8.41713097501839424552372292964719e-03
            4.12641979622127286209121166393743e-01 7.00281686201491698762211512985232e-02 8.41713097501839251080024695284010e-03
            7.00281686201491282428577278551529e-02 5.17329851757723613303596721380018e-01 8.41713097501838383718286706880463e-03
            7.00281686201491421206455356696097e-02 4.12641979622127286209121166393743e-01 8.41713097501838904135329499922591e-03
            7.62594448273204550403647772327531e-01 1.66359403543997558250211454833334e-01 8.88026496784092081449024647099577e-03
            7.62594448273204661425950234843185e-01 7.10461481827977525682626946945675e-02 8.88026496784091387559634256376739e-03
            1.66359403543997613761362686091161e-01 7.62594448273204550403647772327531e-01 8.88026496784091907976677049418868e-03
            1.66359403543997502739060223575507e-01 7.10461481827978219572017337668512e-02 8.88026496784091907976677049418868e-03
            7.10461481827977109348992712511972e-02 7.62594448273204772448252697358839e-01 8.88026496784090346725548670292483e-03
            7.10461481827977803238383103234810e-02 1.66359403543997502739060223575507e-01 8.88026496784091387559634256376739e-03
            6.40898402664594746269699498952832e-01 2.92115957257160496762793400193914e-01 9.69576667591224086728551867508941e-03
            6.40898402664594857292001961468486e-01 6.69856400782446320674168305231433e-02 9.69576667591222698949771086063265e-03
            2.92115957257160496762793400193914e-01 6.40898402664594746269699498952832e-01 9.69576667591223392839161476786103e-03
            2.92115957257160441251642168936087e-01 6.69856400782447292119314852243406e-02 9.69576667591224433673247062870360e-03
            6.69856400782447847230827164821676e-02 6.40898402664594746269699498952832e-01 9.69576667591225127562637453593197e-03
            6.69856400782447014563558695954271e-02 2.92115957257160441251642168936087e-01 9.69576667591223392839161476786103e-03
            5.57643332782426282001608797145309e-01 2.81112663730934020378526838612743e-01 1.44404785842972300385911310627307e-02
            5.57643332782426170979306334629655e-01 1.61244003486639475575259439210640e-01 1.44404785842972300385911310627307e-02
            2.81112663730934131400829301128397e-01 5.57643332782426282001608797145309e-01 1.44404785842972317733146070395378e-02
            2.81112663730934075889678069870570e-01 1.61244003486639669864288748613035e-01 1.44404785842972196302502752018881e-02
            1.61244003486639697619864364241948e-01 5.57643332782426170979306334629655e-01 1.44404785842972144260798472714669e-02
            1.61244003486639558841986286097381e-01 2.81112663730934242423131763644051e-01 1.44404785842972178955267992250810e-02
            ];
        
    case  19
        
        % ALG. DEG.:   19
        % PTS CARD.:   78
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 8.5e-16
        
        xyw=[
            1.19057700723167961959214622424952e-02 1.19057700723167684403458466135817e-02 8.92746659281794772977658425361369e-04
            1.19057700723167302764293751238256e-02 9.76188459855366508222118682169821e-01 8.92746659281788376184840760885209e-04
            9.76188459855366397199816219654167e-01 1.19057700723167441542171829382823e-02 8.92746659281792929833965200003831e-04
            4.66749672319444697254908760442049e-01 4.66749672319444475210303835410741e-01 3.60804761419788880619230475588211e-03
            4.66749672319444475210303835410741e-01 6.65006553611108830459386354050366e-02 3.60804761419793867949223908908607e-03
            6.65006553611110773349679448074312e-02 4.66749672319444475210303835410741e-01 3.60804761419790571974619552975128e-03
            5.44172191454611181571365818854247e-02 5.44172191454610001959402154625423e-02 4.17451226574730525215040088937712e-03
            5.44172191454610695848792545348260e-02 8.91165561709077902463604914373718e-01 4.17451226574729744589475899374520e-03
            8.91165561709077791441302451858064e-01 5.44172191454611250960304857926531e-02 4.17451226574729744589475899374520e-03
            1.49494614598017633300131024043367e-01 1.49494614598017772078009102187934e-01 9.43146227004719624187334403586647e-03
            1.49494614598017494522252945898799e-01 7.01010770803965010955494108202402e-01 9.43146227004718583353248817502390e-03
            7.01010770803965010955494108202402e-01 1.49494614598017494522252945898799e-01 9.43146227004718236408553622140971e-03
            2.45242132245316490912756535180961e-01 2.45242132245316490912756535180961e-01 1.48663889962765034813418907333471e-02
            2.45242132245316657446210228954442e-01 5.09515735509366574085277079575462e-01 1.48663889962764861341071309652762e-02
            5.09515735509366574085277079575462e-01 2.45242132245316685201785844583355e-01 1.48663889962764809299367030348549e-02
            3.74619062916066836166351095016580e-01 3.74619062916066891677502326274407e-01 1.59498745916793645271969381838062e-02
            3.74619062916066725144048632500926e-01 2.50761874167866383178449041224667e-01 1.59498745916793645271969381838062e-02
            2.50761874167866327667297809966840e-01 3.74619062916066725144048632500926e-01 1.59498745916793714660908420910346e-02
            9.33764288367549766256558996246895e-01 5.69009106573930140138273259253765e-02 1.40844251469181172380096267460203e-03
            9.33764288367549544211954071215587e-01 9.33480097505729258799966885362664e-03 1.40844251469181887953530107893130e-03
            5.69009106573929862582517102964630e-02 9.33764288367549766256558996246895e-01 1.40844251469181389220530764561090e-03
            5.69009106573932499362200587711413e-02 9.33480097505718503514415829158679e-03 1.40844251469182169846094954124283e-03
            9.33480097505724401574234150302800e-03 9.33764288367549766256558996246895e-01 1.40844251469181541008834912531711e-03
            9.33480097505720932127282196688611e-03 5.69009106573930556471907493687468e-02 1.40844251469181410904574214271179e-03
            8.65595106103016709298003661388066e-01 1.21539313278482469238284124912752e-01 2.38936846856461224467005344251902e-03
            8.65595106103016709298003661388066e-01 1.28655806185009307512912002380290e-02 2.38936846856460140264832858747468e-03
            1.21539313278482413727132893654925e-01 8.65595106103016709298003661388066e-01 2.38936846856461267835092243672079e-03
            1.21539313278482996594220821862109e-01 1.28655806185009706499311477045921e-02 2.38936846856462785718133723378287e-03
            1.28655806185008856484808248410445e-02 8.65595106103016709298003661388066e-01 2.38936846856461050994657746571193e-03
            1.28655806185009168735033924235722e-02 1.21539313278482399849345085840469e-01 2.38936846856461441307439841352789e-03
            7.80245213926672342985568775475258e-01 2.08294353280180810639166111286613e-01 3.00071243150817944592145813942352e-03
            7.80245213926672898097081088053528e-01 1.14604327931468758655642048438494e-02 3.00071243150817120598494724958982e-03
            2.08294353280180616350136801884219e-01 7.80245213926672454007871237990912e-01 3.00071243150818248168754109883594e-03
            2.08294353280181088194922267575748e-01 1.14604327931468602530529210525856e-02 3.00071243150816253236756736555435e-03
            1.14604327931468619877763970293927e-02 7.80245213926672565030173700506566e-01 3.00071243150817857855972015101997e-03
            1.14604327931468151502425456556011e-02 2.08294353280180505327834339368565e-01 3.00071243150817467543189920320401e-03
            5.47138472365763073312905362399761e-01 4.41297885394377531476095555262873e-01 3.23565844196336801011359973756498e-03
            5.47138472365763073312905362399761e-01 1.15636422398593518429121829171891e-02 3.23565844196339142888052542446076e-03
            4.41297885394377475964944324005046e-01 5.47138472365763184335207824915415e-01 3.23565844196336757643273074336321e-03
            4.41297885394377087386885705200257e-01 1.15636422398593483734652309635749e-02 3.23565844196336584170925476655611e-03
            1.15636422398593657206999907316458e-02 5.47138472365763073312905362399761e-01 3.23565844196337278060315867378449e-03
            1.15636422398593310262304711955039e-02 4.41297885394377475964944324005046e-01 3.23565844196336627539012376075789e-03
            6.65889139137802787971054385707248e-01 3.23322608813239320557642031417345e-01 3.38354969654410062812610604510155e-03
            6.65889139137803343082566698285518e-01 1.07882520489579938199886655070259e-02 3.38354969654410930174348592913702e-03
            3.23322608813239098513037106386037e-01 6.65889139137802898993356848222902e-01 3.38354969654410539861566498132106e-03
            3.23322608813239320557642031417345e-01 1.07882520489579730033069537853407e-02 3.38354969654408588297656024224125e-03
            1.07882520489579695338600018317265e-02 6.65889139137802898993356848222902e-01 3.38354969654410323021132001031219e-03
            1.07882520489579521866252420636556e-02 3.23322608813239209535339568901691e-01 3.38354969654409715867915409148736e-03
            7.97030463292363489102854146040045e-01 1.39071275726620063872829291540256e-01 7.12773874090821824012209617649205e-03
            7.97030463292363711147459071071353e-01 6.38982609810165302910434093064396e-02 7.12773874090821130122819226926367e-03
            1.39071275726619980606102444653516e-01 7.97030463292363489102854146040045e-01 7.12773874090821824012209617649205e-03
            1.39071275726620174895131754055910e-01 6.38982609810166551911336796365504e-02 7.12773874090822778110121404893107e-03
            6.38982609810165441688312171208963e-02 7.97030463292363600125156608555699e-01 7.12773874090821477067514422287786e-03
            6.38982609810166274355580640076369e-02 1.39071275726619952850526829024602e-01 7.12773874090822344429252410691333e-03
            6.88298973534611890912060516711790e-01 2.54674408859317524811416433294653e-01 8.00189652345183015380580826558798e-03
            6.88298973534612446023572829290060e-01 5.70266176060704593764327796634461e-02 8.00189652345181107184757252070995e-03
            2.54674408859317413789113970778999e-01 6.88298973534612001934362979227444e-01 8.00189652345182668435885631197380e-03
            2.54674408859317469300265202036826e-01 5.70266176060703552930242210550205e-02 8.00189652345180760240062056709576e-03
            5.70266176060704593764327796634461e-02 6.88298973534612223978967904258752e-01 8.00189652345181974546495240474542e-03
            5.70266176060703622319181249622488e-02 2.54674408859317413789113970778999e-01 8.00189652345180760240062056709576e-03
            5.57143388414403584185663476091577e-01 3.85518010456119386564211026779958e-01 8.18104584356584008109081196380430e-03
            5.57143388414403917252570863638539e-01 5.73386011294770847612767283862922e-02 8.18104584356584355053776391741849e-03
            3.85518010456119386564211026779958e-01 5.57143388414403584185663476091577e-01 8.18104584356585395887861977826105e-03
            3.85518010456119386564211026779958e-01 5.73386011294770708834889205718355e-02 8.18104584356582793802648012615464e-03
            5.73386011294770500668072088501503e-02 5.57143388414403695207965938607231e-01 8.18104584356585742832557173187524e-03
            5.73386011294770292501254971284652e-02 3.85518010456119386564211026779958e-01 8.18104584356585048943166782464687e-03
            6.12834567107272132346906801103614e-01 2.52827382426030478068668116975459e-01 1.02443174346838750121690964078880e-02
            6.12834567107272132346906801103614e-01 1.34338050466697306317698235034186e-01 1.02443174346838836857864762919235e-02
            2.52827382426030478068668116975459e-01 6.12834567107272132346906801103614e-01 1.02443174346838767468925723846951e-02
            2.52827382426030589090970579491113e-01 1.34338050466696973250790847487224e-01 1.02443174346838628691047645702383e-02
            1.34338050466697250806547003776359e-01 6.12834567107272132346906801103614e-01 1.02443174346838767468925723846951e-02
            1.34338050466696973250790847487224e-01 2.52827382426030811135575504522421e-01 1.02443174346838576649343366398170e-02
            4.92042973154920293410441445303150e-01 3.66154971066125212164621416377486e-01 1.18990870388364939486125493317559e-02
            4.92042973154920182388138982787495e-01 1.41802055778954466669361522690451e-01 1.18990870388364939486125493317559e-02
            3.66154971066125267675772647635313e-01 4.92042973154920348921592676560977e-01 1.18990870388364922138890733549488e-02
            3.66154971066125101142318953861832e-01 1.41802055778954577691663985206105e-01 1.18990870388364835402716934709133e-02
            1.41802055778954522180512753948278e-01 4.92042973154920237899290214045323e-01 1.18990870388364766013777895636849e-02
            1.41802055778954522180512753948278e-01 3.66154971066125212164621416377486e-01 1.18990870388364731319308376100707e-02
            ];
        
    case  20
        
        % ALG. DEG.:   20
        % PTS CARD.:   91
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.3e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 1.39707282352902004834982463421511e-02
            8.57077238529909204989287729858916e-03 8.57077238529917878606667613894388e-03 4.95279077919163145636571865537690e-04
            8.57077238529914756104410855641618e-03 9.82858455229401739572381302423310e-01 4.95279077919165422461134085097001e-04
            9.82858455229401850594683764938964e-01 8.57077238529911286657458902027429e-03 4.95279077919164121418527102491680e-04
            5.15305405051263848426046365602815e-02 5.15305405051261489202119037145167e-02 3.20424682525371432723071407622228e-03
            5.15305405051262738203021740446275e-02 8.96938918989747424603820036281832e-01 3.20424682525371172514550011101164e-03
            8.96938918989747424603820036281832e-01 5.15305405051263085147716935807694e-02 3.20424682525371866403940401824002e-03
            4.94659340413900083799347839885741e-01 4.94659340413900083799347839885741e-01 3.48200600375842906145096300463138e-03
            4.94659340413900083799347839885741e-01 1.06813191721997491345774733417784e-02 3.48200600375842168887619010320122e-03
            1.06813191721997803596000409243061e-02 4.94659340413900083799347839885741e-01 3.48200600375842949513183199883315e-03
            4.37325812915608291842062271825853e-01 4.37325812915608291842062271825853e-01 8.33670305949367820841366949480289e-03
            4.37325812915608291842062271825853e-01 1.25348374168783277537997378203727e-01 8.33670305949366086117890972673194e-03
            1.25348374168783444071451071977208e-01 4.37325812915608069797457346794545e-01 8.33670305949367473896671754118870e-03
            1.32293995381116469278737213244312e-01 1.32293995381116663567766522646707e-01 8.37571939581808165564691393001340e-03
            1.32293995381116608056615291388880e-01 7.35412009237766839397920648480067e-01 8.37571939581807992092343795320630e-03
            7.35412009237766839397920648480067e-01 1.32293995381116497034312828873226e-01 8.37571939581808512509386588362759e-03
            2.15238824344608503658093923149863e-01 2.15238824344608420391367076263123e-01 1.12128489726990750174895339341674e-02
            2.15238824344608559169245154407690e-01 5.69522351310782992683812153700273e-01 1.12128489726990906300008177254313e-02
            5.69522351310782992683812153700273e-01 2.15238824344608392635791460634209e-01 1.12128489726990802216599618645887e-02
            9.43408146097875710545110905513866e-01 4.66217699188314349534323355328524e-02 1.36692727975725250577632152726437e-03
            9.43408146097875377478203517966904e-01 9.97008398329295511541836560809315e-03 1.36692727975725727626588046348388e-03
            4.66217699188315945479921253991051e-02 9.43408146097875488500505980482558e-01 1.36692727975725489102110099537413e-03
            4.66217699188313447478115847388835e-02 9.97008398329291521677841814152998e-03 1.36692727975724968685067306495284e-03
            9.97008398329294991124793767767187e-03 9.43408146097875488500505980482558e-01 1.36692727975725445734023200117235e-03
            9.97008398329293776818360584002221e-03 4.66217699188316778147189722858457e-02 1.36692727975725814362761845188743e-03
            7.95199869823192462270355918008136e-01 1.94613925044525570839937245182227e-01 1.93561846434583246200078487930796e-03
            7.95199869823192462270355918008136e-01 1.01862051322818593368513262475972e-02 1.93561846434582140313862552716273e-03
            1.94613925044525598595512860811141e-01 7.95199869823192462270355918008136e-01 1.93561846434583159463904689090441e-03
            1.94613925044525598595512860811141e-01 1.01862051322819217868964614126526e-02 1.93561846434583614828817133002303e-03
            1.01862051322819582160894569256016e-02 7.95199869823192462270355918008136e-01 1.93561846434583463040512985031683e-03
            1.01862051322819096438321295750029e-02 1.94613925044525598595512860811141e-01 1.93561846434582942623470191989554e-03
            7.24428132024519610965285210113507e-01 2.65496289314309019680138135299785e-01 2.07811690027082020953974605959047e-03
            7.24428132024519610965285210113507e-01 1.00755786611713450684479909114089e-02 2.07811690027082151058235304219579e-03
            2.65496289314309075191289366557612e-01 7.24428132024519610965285210113507e-01 2.07811690027082151058235304219579e-03
            2.65496289314309075191289366557612e-01 1.00755786611713745587470825171295e-02 2.07811690027081500536931812916919e-03
            1.00755786611713173128723752824953e-02 7.24428132024519610965285210113507e-01 2.07811690027082194426322203639756e-03
            1.00755786611713762934705584939366e-02 2.65496289314309075191289366557612e-01 2.07811690027082584739104298421353e-03
            8.75117681769159161397908519575140e-01 1.14200622336593365924173326675373e-01 2.15687460499512477507533247944593e-03
            8.75117681769159050375606057059485e-01 1.06816958942474518612364420278027e-02 2.15687460499512607611793946205125e-03
            1.14200622336593421435324557933200e-01 8.75117681769159161397908519575140e-01 2.15687460499511913722403555482288e-03
            1.14200622336593227146295248530805e-01 1.06816958942474882904294375407517e-02 2.15687460499512867820315342726190e-03
            1.06816958942474431876190621437672e-02 8.75117681769159161397908519575140e-01 2.15687460499512087194751153162997e-03
            1.06816958942474189014903984684679e-02 1.14200622336593490824263597005483e-01 2.15687460499511566777708360120869e-03
            6.21242229278482605003830485657090e-01 3.69462178934587270884293275230448e-01 2.88718927896879735614521500508545e-03
            6.21242229278482493981528023141436e-01 9.29559178693014492855795083414705e-03 2.88718927896879648878347701668190e-03
            3.69462178934587437417746969003929e-01 6.21242229278482382959225560625782e-01 2.88718927896879432037913204567303e-03
            3.69462178934587326395444506488275e-01 9.29559178693014492855795083414705e-03 2.88718927896879388669826305147126e-03
            9.29559178693013798966404692691867e-03 6.21242229278482382959225560625782e-01 2.88718927896879432037913204567303e-03
            9.29559178693011890770581118204063e-03 3.69462178934587381906595737746102e-01 2.88718927896879258565565606886594e-03
            8.23284162466894064991151935828384e-01 1.21432174389561192695552449549723e-01 4.99989919370117351371529679227024e-03
            8.23284162466894287035756860859692e-01 5.52836631435446521076748638279241e-02 4.99989919370117698316224874588443e-03
            1.21432174389561137184401218291896e-01 8.23284162466894287035756860859692e-01 4.99989919370118131997093868790216e-03
            1.21432174389561234328915872993093e-01 5.52836631435447006799321911785228e-02 4.99989919370118218733267667630571e-03
            5.52836631435446382298870560134674e-02 8.23284162466894287035756860859692e-01 4.99989919370118045260920069949861e-03
            5.52836631435445410853724013122701e-02 1.21432174389561012284310947961785e-01 4.99989919370117785052398673428797e-03
            7.35502973872642962227530460950220e-01 2.11973966375945732654173525588703e-01 5.76931548784120461909052934856845e-03
            7.35502973872642962227530460950220e-01 5.25230597514112565460386861104780e-02 5.76931548784121589479312319781457e-03
            2.11973966375945843676475988104357e-01 7.35502973872642851205227998434566e-01 5.76931548784121155798443325579683e-03
            2.11973966375945788165324756846530e-01 5.25230597514113189960838212755334e-02 5.76931548784120375172879136016491e-03
            5.25230597514113467516594369044469e-02 7.35502973872642851205227998434566e-01 5.76931548784121589479312319781457e-03
            5.25230597514113745072350525333604e-02 2.11973966375945788165324756846530e-01 5.76931548784122283368702710504294e-03
            5.24604158534802356150805735524045e-01 4.22476993945888334991423107567243e-01 5.92421552443886736960543260011036e-03
            5.24604158534802356150805735524045e-01 5.29188475193092394688321178364276e-02 5.92421552443885782862631472767134e-03
            4.22476993945888390502574338825070e-01 5.24604158534802356150805735524045e-01 5.92421552443886129807326668128553e-03
            4.22476993945888390502574338825070e-01 5.29188475193092117132565022075141e-02 5.92421552443886390015848064649617e-03
            5.29188475193092811021955412797979e-02 5.24604158534802134106200810492737e-01 5.92421552443886303279674265809263e-03
            5.29188475193092811021955412797979e-02 4.22476993945888390502574338825070e-01 5.92421552443884395083850691321459e-03
            6.30873630515176087030226881324779e-01 3.19958418028188196213790206456906e-01 6.05389449191200915839639051796439e-03
            6.30873630515175976007924418809125e-01 4.91679514566356820615133926821727e-02 6.05389449191201089311986649477149e-03
            3.19958418028188418258395131488214e-01 6.30873630515175976007924418809125e-01 6.05389449191200482158770057594666e-03
            3.19958418028188251724941437714733e-01 4.91679514566356473670438731460308e-02 6.05389449191200829103465252956084e-03
            4.91679514566356334892560653315741e-02 6.30873630515175976007924418809125e-01 6.05389449191201609729029442519277e-03
            4.91679514566356612448316809604876e-02 3.19958418028188529280697594003868e-01 6.05389449191200915839639051796439e-03
            6.45556379414073733258305765048135e-01 2.34468421561538770125920905229577e-01 8.65015178774573671727221579885736e-03
            6.45556379414073733258305765048135e-01 1.19975199024387316204531828134350e-01 8.65015178774573845199569177566445e-03
            2.34468421561538908903798983374145e-01 6.45556379414073733258305765048135e-01 8.65015178774572630893135993801479e-03
            2.34468421561538742370345289600664e-01 1.19975199024387357837895251577720e-01 8.65015178774572977837831189162898e-03
            1.19975199024387454982409906278917e-01 6.45556379414073733258305765048135e-01 8.65015178774573498254873982205027e-03
            1.19975199024387468860197714093374e-01 2.34468421561538964414950214631972e-01 8.65015178774572804365483591482189e-03
            5.39652455990818680220399983227253e-01 3.37729237107069868439879201105214e-01 8.83375676654125745190793139727248e-03
            5.39652455990818458175795058195945e-01 1.22618306902111423584145200038620e-01 8.83375676654124010467317162920153e-03
            3.37729237107069923951030432363041e-01 5.39652455990818680220399983227253e-01 8.83375676654125224773750346685119e-03
            3.37729237107069923951030432363041e-01 1.22618306902111354195206160966336e-01 8.83375676654125745190793139727248e-03
            1.22618306902111340317418353151879e-01 5.39652455990818680220399983227253e-01 8.83375676654126612552531128130795e-03
            1.22618306902111395828569584409706e-01 3.37729237107069923951030432363041e-01 8.83375676654122969633231576835897e-03
            4.51386500463365036850404976576101e-01 3.28819489995002267601620360437664e-01 1.27955171794624875808432662438463e-02
            4.51386500463364925828102514060447e-01 2.19794009541632556770096584841667e-01 1.27955171794624841113963142902321e-02
            3.28819489995002489646225285468972e-01 4.51386500463364925828102514060447e-01 1.27955171794624893155667422206534e-02
            3.28819489995002434135074054211145e-01 2.19794009541632473503369737954927e-01 1.27955171794624806419493623366179e-02
            2.19794009541632556770096584841667e-01 4.51386500463365036850404976576101e-01 1.27955171794625031933545500351102e-02
            2.19794009541632640036823431728408e-01 3.28819489995002323112771591695491e-01 1.27955171794624979891841221046889e-02
            ];
        
    case  22
        
        % ALG. DEG.:   22
        % PTS CARD.:  105
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.7e-16
        
        xyw=[
            5.81913467082437706190400916739236e-03 5.81913467082493390813979772246967e-03 2.34318517028995749844344498313831e-04
            5.81913467082731741819578985541739e-03 9.88361730658345760680560943001183e-01 2.34318517029105931890123337701937e-04
            9.88361730658347092948190493189031e-01 5.81913467082685945119813197834446e-03 2.34318517029068662440444148487018e-04
            3.70242529014878085646955696574878e-02 3.70242529015019708471534443106066e-02 1.66953744906592075128437535624926e-03
            3.70242529014817092769540352037438e-02 9.25951494197024382870608860685024e-01 1.66953744906604608505551468056183e-03
            9.25951494197024382870608860685024e-01 3.70242529014770810347201290824160e-02 1.66953744906596108360519181701420e-03
            1.00139964243313464575813043211383e-01 1.00139964243318779768543436148320e-01 5.04726064745936609662324556779822e-03
            1.00139964243327175830167163894657e-01 7.99720071513356778325487539405003e-01 5.04726064745925073751209311012644e-03
            7.99720071513358554682326939655468e-01 1.00139964243326967663350046677806e-01 5.04726064745919435899912386389587e-03
            4.77283361444081621893786859800457e-01 4.77283361444104714532699063056498e-01 5.78994206463942420043400360896158e-03
            4.77283361444093778835906505264575e-01 4.54332771118138301069677709165262e-02 5.78994206463894194730768205658933e-03
            4.54332771118135872456811341635330e-02 4.77283361444090392655681398537126e-01 5.78994206463844494903181470135678e-03
            1.87840033887680113444673679623520e-01 1.87840033887668983458851812429202e-01 8.63526390138759890324759993518455e-03
            1.87840033887670343482056978245964e-01 6.24319932224627338612776838999707e-01 8.63526390138822860786937951615982e-03
            6.24319932224634666084739365032874e-01 1.87840033887671259416052294000110e-01 8.63526390138817656616510021194699e-03
            4.02759919721537684722534322645515e-01 4.02759919721540182724339729247731e-01 1.10728355683578239682107735575300e-02
            4.02759919721531467473596421768889e-01 1.94480160556923437065179882665689e-01 1.10728355683567605827199997747812e-02
            1.94480160556922188064277179364581e-01 4.02759919721522974267458039321355e-01 1.10728355683568542577877025223643e-02
            2.95819836686845805129308928371756e-01 2.95819836686848303131114334973972e-01 1.20253106664278356513486656353962e-02
            2.95819836686846526774274934723508e-01 4.08360326626306835429147668037331e-01 1.20253106664280663695709705507397e-02
            4.08360326626305503161518117849482e-01 2.95819836686846415751972472207854e-01 1.20253106664280576959535906667043e-02
            9.60353805601490417842569513595663e-01 3.24378361207835266832866238928545e-02 6.86218711192522891166578613564297e-04
            9.60353805601490417842569513595663e-01 7.20835827771566187843754747177627e-03 6.86218711192045083269164251760230e-04
            3.24378361207852336511869850710355e-02 9.60353805601490417842569513595663e-01 6.86218711192404279448908699379217e-04
            3.24378361207898827101026029140485e-02 7.20835827772483769826372679290216e-03 6.86218711192772474506684776685006e-04
            7.20835827771712251560431994334976e-03 9.60353805601490417842569513595663e-01 6.86218711192045733790467743062891e-04
            7.20835827772210117198037337971073e-03 3.24378361207898827101026029140485e-02 6.86218711192471391563385552103682e-04
            9.12769702185145415107569988322211e-01 7.96405637940718197365086439276638e-02 1.09985439787456894776551674652865e-03
            9.12769702185116660331232196767814e-01 7.58973402078859816360933976397973e-03 1.09985439787517219785428768119573e-03
            7.96405637940748173386751318503229e-02 9.12769702185143194661520738009131e-01 1.09985439787448026002780743226594e-03
            7.96405637940890281933903338540404e-02 7.58973402078647225998953018688553e-03 1.09985439787495514057935608320804e-03
            7.58973402078918363278248193637410e-03 9.12769702185126097226941510598408e-01 1.09985439787511777090522890887314e-03
            7.58973402078367068157582764342806e-03 7.96405637940886951264829463070782e-02 1.09985439787461491793763013191665e-03
            8.46930929612685123863968783552991e-01 1.44767660510349893687731537283980e-01 1.56895738507179254191126194228900e-03
            8.46930929612685123863968783552991e-01 8.30140987696372303905612000107794e-03 1.56895738507106807801960712822620e-03
            1.44767660510350892888453699924867e-01 8.46930929612682903417919533239910e-01 1.56895738507192980190629860715035e-03
            1.44767660510350892888453699924867e-01 8.30140987696267700080010598640001e-03 1.56895738507099196702709864581493e-03
            8.30140987696306210941177283757497e-03 8.46930929612694893826585484930547e-01 1.56895738507112640809648684836475e-03
            8.30140987696398324757751652214210e-03 1.44767660510350892888453699924867e-01 1.56895738507119080970553248732813e-03
            7.63800852902573046421252911386546e-01 2.27743920492919527109876298709423e-01 1.99015381152270725423347741411817e-03
            7.63800852902583926606894237920642e-01 8.45522660450583572744331917192540e-03 1.99015381152238806511389768161280e-03
            2.27743920492916557263285426415678e-01 7.63800852902575821978814474277897e-01 1.99015381152270725423347741411817e-03
            2.27743920492911561259674613211246e-01 8.45522660450640992091386749507365e-03 1.99015381152247957177725545818703e-03
            8.45522660450593807612840180354397e-03 7.63800852902583815584591775404988e-01 1.99015381152254939439716352467258e-03
            8.45522660450685921429414548811110e-03 2.27743920492909812658410828589695e-01 1.99015381152259016039884897963930e-03
            6.63853996353000908037245153536787e-01 3.27598731749433658144710079795914e-01 2.33250190153522088745696727585255e-03
            6.63853996353002795416387016302906e-01 8.54727189756490125793764178752099e-03 2.33250190153527769965080551628489e-03
            3.27598731749432103832475604576757e-01 6.63853996353002684394084553787252e-01 2.33250190153523389788303710190576e-03
            3.27598731749430716053694823131082e-01 8.54727189756544596110909850494863e-03 2.33250190153535836429243843781478e-03
            8.54727189756502442330443614082469e-03 6.63853996353001463148757466115057e-01 2.33250190153523476524477509030930e-03
            8.54727189756525514152674105616825e-03 3.27598731749432103832475604576757e-01 2.33250190153541257440106271303648e-03
            5.52855567624573107643470848415745e-01 4.38428235397557652941458172790590e-01 2.53835855579786226834948514863299e-03
            5.52855567624569665952094510430470e-01 8.71619697786918216919627155903072e-03 2.53835855579784925792341532257979e-03
            4.38428235397559484809448804298881e-01 5.52855567624571442308933910680935e-01 2.53835855579787874822250692830039e-03
            4.38428235397557819474911866564071e-01 8.71619697786932962069172958763374e-03 2.53835855579787354405207899787911e-03
            8.71619697786919604698407937348748e-03 5.52855567624573107643470848415745e-01 2.53835855579776078702614050541797e-03
            8.71619697786918390391974753583781e-03 4.38428235397562870989673911026330e-01 2.53835855579786747251991307905428e-03
            8.69853922593163075838162967556855e-01 8.94784746607361763937049659034528e-02 2.94314417090164480092506771313765e-03
            8.69853922593129991192029137891950e-01 4.06676027461178452027112939504150e-02 2.94314417090148433900353985848142e-03
            8.94784746607384245953298318454472e-02 8.69853922593161188459021104790736e-01 2.94314417090144877717228233393598e-03
            8.94784746607328734802067060627451e-02 4.06676027461085748404556738933024e-02 2.94314417090101769838850209737302e-03
            4.06676027461189415479481112924987e-02 8.69853922593127548701374962547561e-01 2.94314417090194317336293572395789e-03
            4.06676027461031347476350106262544e-02 8.94784746607552861075163264104049e-02 2.94314417090106800536930542477876e-03
            7.92000836049181700104782066773623e-01 1.64479570130585700526992809500371e-01 4.12209846528247166042024929311083e-03
            7.92000836049191914156608618213795e-01 4.35195938202235649283622365146584e-02 4.12209846528128597692441914546180e-03
            1.64479570130581009834713768213987e-01 7.92000836049181700104782066773623e-01 4.12209846528231727003088735727943e-03
            1.64479570130567853691871960108983e-01 4.35195938202214555046154487172316e-02 4.12209846528147939859199055945282e-03
            4.35195938202220175550216652027302e-02 7.92000836049181700104782066773623e-01 4.12209846528168149387694185747932e-03
            4.35195938202279017370521785323945e-02 1.64479570130591612464598938458948e-01 4.12209846528172225987862731244604e-03
            6.98032087133783618604354614944896e-01 2.57896711320231286634907519328408e-01 4.96292855328392575053886304203843e-03
            6.98032087133790946076317140978063e-01 4.40712015459795644622964516656793e-02 4.96292855328353196830981630682800e-03
            2.57896711320215854534865229652496e-01 6.98032087133799494793606754683424e-01 4.96292855328344523213601746647328e-03
            2.57896711320220961560778505372582e-01 4.40712015459825898200385552172520e-02 4.96292855328420243893328134276999e-03
            4.40712015459804318240344400692265e-02 6.98032087133772738418713288410800e-01 4.96292855328361610239840118197208e-03
            4.40712015459831588093386756099790e-02 2.57896711320235505482401094923262e-01 4.96292855328375661499995530334672e-03
            5.90921636898512403490713040810078e-01 3.64207294093871758455804865661776e-01 5.55749662394772886431226766035252e-03
            5.90921636898516511315904153889278e-01 4.48710690076145682358976785053528e-02 5.55749662394731946957193713387824e-03
            3.64207294093850664218336987687508e-01 5.90921636898534607951205543940887e-01 5.55749662394743482868308959155002e-03
            3.64207294093867151030252671262133e-01 4.48710690076162890815858474979905e-02 5.55749662394774100737659949800218e-03
            4.48710690076150608973648559185676e-02 5.90921636898503632728818502073409e-01 5.55749662394695951445067194640615e-03
            4.48710690076153176364393004860176e-02 3.64207294093872813167678259560489e-01 5.55749662394721018199295059503129e-03
            7.13756998577414591977685631718487e-01 1.81577875343951838926770392390608e-01 6.55379828019880036199529271812025e-03
            7.13756998577440460174159397865878e-01 1.04665126078613612836676338702091e-01 6.55379828019890184331863736133528e-03
            1.81577875343954003861668411445862e-01 7.13756998577404933037371392856585e-01 6.55379828019910827541227860137951e-03
            1.81577875343944039610022400665912e-01 1.04665126078612918947285947979253e-01 6.55379828019882898493264633543731e-03
            1.04665126078611864235412554080540e-01 7.13756998577441792441788948053727e-01 6.55379828019878128003705697324222e-03
            1.04665126078623660355049196368782e-01 1.81577875343950506659140842202760e-01 6.55379828019910307124185067095823e-03
            6.14456089155397067180786052631447e-01 2.79145603763534810859425761009334e-01 7.76522704867182653187818175410939e-03
            6.14456089155397067180786052631447e-01 1.06398307081061002654642777542904e-01 7.76522704867228016206714968916458e-03
            2.79145603763536809260870086291106e-01 6.14456089155397067180786052631447e-01 7.76522704867172938736352705291210e-03
            2.79145603763538918684616874088533e-01 1.06398307081066303969585362665384e-01 7.76522704867227235581150779353266e-03
            1.06398307081062487577938213689777e-01 6.14456089155397067180786052631447e-01 7.76522704867215265989166539384314e-03
            1.06398307081065290891075392210041e-01 2.79145603763543193043261680941214e-01 7.76522704867208413831436430996291e-03
            5.03163853150141560099939397332491e-01 3.88410709745710069906721173538244e-01 8.56759707932744016267712083845254e-03
            5.03163853150135009784094108908903e-01 1.08425437104147454059344113375118e-01 8.56759707932695964427427526288739e-03
            3.88410709745712234841619192593498e-01 5.03163853150140560899217234691605e-01 8.56759707932742281544236107038159e-03
            3.88410709745712956486585198945249e-01 1.08425437104149285927334744883410e-01 8.56759707932692148035780377313131e-03
            1.08425437104147551203858768076316e-01 5.03163853150138340453167984378524e-01 8.56759707932685209141876470084753e-03
            1.08425437104147801004039308736537e-01 3.88410709745717064311776312024449e-01 8.56759707932689199005871216741070e-03
            5.18446604434300351016418062499724e-01 2.89296348423914995162675722895074e-01 1.04077639415439439363186835407760e-02
            5.18446604434288582652357035840396e-01 1.92257047141778936172329395049019e-01 1.04077639415432465774813408643240e-02
            2.89296348423913718406197403965052e-01 5.18446604434304458841609175578924e-01 1.04077639415439231196369718190908e-02
            2.89296348423917604186783592012944e-01 1.92257047141783599109032820706489e-01 1.04077639415430661662398392763862e-02
            1.92257047141780823551471257815138e-01 5.18446604434278701667437871947186e-01 1.04077639415432344344170090266744e-02
            1.92257047141780823551471257815138e-01 2.89296348423907889735318121893215e-01 1.04077639415431806579892537456544e-02
            ];
        
    case  24
        
        % ALG. DEG.:   24
        % PTS CARD.:  120
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.2e-16
        
        xyw=[
            4.68044773847062830357534934933028e-03 4.68044773847102295316613407294426e-03 1.68957158836241755177354217920538e-04
            4.68044773847029350194448582556106e-03 9.90639104523059477180879639490740e-01 1.68957158836215110908965386649072e-04
            9.90639104523057256734830389177660e-01 4.68044773847173939396171249427425e-03 1.68957158836265770255474771843751e-04
            3.58515749284085347148298694719415e-02 3.58515749284044130118509485782852e-02 1.21250442126557525686358918193264e-03
            3.58515749284033513610836507723434e-02 9.28296850143190410697968673048308e-01 1.21250442126547724498719649233180e-03
            9.28296850143188079229616960219573e-01 3.58515749284062726354171957154904e-02 1.21250442126554554972406307911115e-03
            4.96163793331061475644361280501471e-01 4.96163793331063862623864224588033e-01 2.08111478604418830007372065438176e-03
            4.96163793331063363023503143267590e-01 7.67241333787459928172935974544089e-03 2.08111478604417312124330585731968e-03
            7.67241333787459494492066980342315e-03 4.96163793331062530356234674400184e-01 2.08111478604417485596678183412678e-03
            8.10973021217381573322313670360018e-02 8.10973021217408635008538908550690e-02 3.91437817646203382870861986475575e-03
            8.10973021217391010218022984190611e-02 8.37805395756522242045605253224494e-01 3.91437817646195836823741487364714e-03
            8.37805395756521242844883090583608e-01 8.10973021217403500227050017201691e-02 3.91437817646202255300602601550963e-03
            1.67282792916414513317846513018594e-01 1.67282792916416844786198225847329e-01 6.80908259294159094732190595777865e-03
            1.67282792916414041473061047327064e-01 6.65434414167170640297399586415850e-01 6.80908259294153630353241268835518e-03
            6.65434414167167864739838023524499e-01 1.67282792916414596584573359905335e-01 6.80908259294150507850984510582748e-03
            4.51971022222129203971263677885872e-01 4.51971022222131368906161696941126e-01 7.29831918563668153449031095192368e-03
            4.51971022222132146062278934550704e-01 9.60579555557396630449673580187664e-02 7.29831918563664510529731543897469e-03
            9.60579555557397879450576283488772e-02 4.51971022222131646461917853230261e-01 7.29831918563656791010263447105899e-03
            2.62525142845062731122851573672961e-01 2.62525142845060122098743704555091e-01 1.00309136241977531894820074853669e-02
            2.62525142845060344143348629586399e-01 4.74949714309878590068336734475452e-01 1.00309136241977531894820074853669e-02
            4.74949714309880810514385984788532e-01 2.62525142845060788232558479649015e-01 1.00309136241977445158646276013314e-02
            3.67318860036336902741282983697602e-01 3.67318860036336625185526827408466e-01 1.06711873022694095358708921139623e-02
            3.67318860036338457053517458916758e-01 2.65362279927327138207004963987856e-01 1.06711873022694025969769882067339e-02
            2.65362279927327859851970970339607e-01 3.67318860036335959051712052314542e-01 1.06711873022693956580830842995056e-02
            9.23455229698260882109650538041024e-01 7.19187624498168059927749595772184e-02 6.30270027278618649030361709151293e-04
            9.23455229698260882109650538041024e-01 4.62600785192418626629029532182358e-03 6.30270027278772714159071899331366e-04
            7.19187624498129618455521949726972e-02 9.23455229698263657667212100932375e-01 6.30270027278725117683699785686713e-04
            7.19187624498160982255967610399239e-02 4.62600785192388355704373736898560e-03 6.30270027278787567728834950742112e-04
            4.62600785192390263900197311386364e-03 9.23455229698260882109650538041024e-01 6.30270027278772605738854650780922e-04
            4.62600785192329375106190525457350e-03 7.19187624498160982255967610399239e-02 6.30270027278690531634397498095268e-04
            9.63258241935609049733102438040078e-01 2.93101710973737335230726586132732e-02 6.53845666855053611089820808643935e-04
            9.63258241935611159156849225837504e-01 7.43158696701679954282893092454287e-03 6.53845666855017507157477041346283e-04
            2.93101710973702675455676569526986e-02 9.63258241935612713469083701056661e-01 6.53845666855010893524224879769235e-04
            2.93101710973716553243484383983741e-02 7.43158696701670066359080024653849e-03 6.53845666855013604029656093530321e-04
            7.43158696701625657438095018392232e-03 9.63258241935612602446781238541007e-01 6.53845666854974572751446615370696e-04
            7.43158696701714475280065030915466e-03 2.93101710973726579945175529928747e-02 6.53845666855066838356325131798030e-04
            8.65062183147061403154509662272176e-01 1.27671748918951943885247146681650e-01 1.16094094394982114831771369267699e-03
            8.65062183147064511778978612710489e-01 7.26606793398609871609261645630795e-03 1.16094094394977149185821385657391e-03
            1.27671748918951111217978677814244e-01 8.65062183147062846444441674975678e-01 1.16094094394981876307293422456723e-03
            1.27671748918952387974456996744266e-01 7.26606793398645606912866767856940e-03 1.16094094394979664534861552027678e-03
            7.26606793398613688000908794606403e-03 8.65062183147062735422139212460024e-01 1.16094094394976433612387545224465e-03
            7.26606793398584457910338585406862e-03 1.27671748918951111217978677814244e-01 1.16094094394972834061174893349744e-03
            7.93149355629101493470045625144849e-01 1.99376854580636947433802674822800e-01 1.55806520214146499575535997905718e-03
            7.93149355629101493470045625144849e-01 7.47378979026149404402135090208503e-03 1.55806520214150142494835549200616e-03
            1.99376854580639029101973846991314e-01 7.93149355629099384046298837347422e-01 1.55806520214148451139446471813699e-03
            1.99376854580637002944953906080627e-01 7.47378979026163889343159496547742e-03 1.55806520214146564627666347035984e-03
            7.47378979026150185027699279771696e-03 7.93149355629101493470045625144849e-01 1.55806520214145740634015258052614e-03
            7.47378979026160506632381341773907e-03 1.99376854580638029901251684350427e-01 1.55806520214149773866096904129108e-03
            7.03167527087426136311876234685769e-01 2.89355487561615498659506329204305e-01 1.85891476622380654760979812323285e-03
            7.03167527087424581999641759466613e-01 7.47698535095845870368513885750872e-03 1.85891476622382758113194434201887e-03
            2.89355487561616608882530954360845e-01 7.03167527087424915066549147013575e-01 1.85891476622380134343937019281157e-03
            2.89355487561615443148355097946478e-01 7.47698535095848038772858856759740e-03 1.85891476622382649692977185651444e-03
            7.47698535095843094810952322859521e-03 7.03167527087425692222666384623153e-01 1.85891476622382259380195090869847e-03
            7.47698535095859314475452706005854e-03 2.89355487561616941949438341907808e-01 1.85891476622384818097322156660312e-03
            8.94987050429267849693815151113085e-01 7.57231791336120396707087820686866e-02 1.89839101145379768667553754823984e-03
            8.94987050429267738671512688597431e-01 2.92897704371233684461639512619513e-02 1.89839101145373024930040894986405e-03
            7.57231791336077236787005517726357e-02 8.94987050429270736273679176520091e-01 1.89839101145378380888772973378309e-03
            7.57231791336120257929209742542298e-02 2.92897704371226676178796566318852e-02 1.89839101145366823293614277901042e-03
            2.92897704371223970010174042499784e-02 8.94987050429267849693815151113085e-01 1.89839101145368211072395059346718e-03
            2.92897704371209051388280641958772e-02 7.57231791336085285903934050111275e-02 1.89839101145373653767300936578977e-03
            6.01693105035377984179945087817032e-01 3.90708026101555361275785571706365e-01 2.02984327382668130676512951993118e-03
            6.01693105035377984179945087817032e-01 7.59886886306671439222926167644800e-03 2.02984327382666049008341779824605e-03
            3.90708026101554140030458484034170e-01 6.01693105035379094402969712973572e-01 2.02984327382666179112602478085137e-03
            3.90708026101555416786936802964192e-01 7.59886886306669877971797788518415e-03 2.02984327382669128142511638657197e-03
            7.59886886306669357554754995476287e-03 6.01693105035377984179945087817032e-01 2.02984327382667870467991555472054e-03
            7.59886886306676036240137506183601e-03 3.90708026101555416786936802964192e-01 2.02984327382667436787122561270280e-03
            8.22015697438519876705242950265529e-01 1.39604852336159324099895684412331e-01 3.21021337904420136361016702153393e-03
            8.22015697438525094753458688501269e-01 3.83794502253187383433719048753119e-02 3.21021337904415322503370866513706e-03
            1.39604852336156076697548655829451e-01 8.22015697438525094753458688501269e-01 3.21021337904415192399110168253173e-03
            1.39604852336156298742153580860759e-01 3.83794502253202440833490527438698e-02 3.21021337904414064828850783328562e-03
            3.83794502253187591600536165969970e-02 8.22015697438525094753458688501269e-01 3.21021337904411376007463019277566e-03
            3.83794502253179126149973399151349e-02 1.39604852336156465275607274634240e-01 3.21021337904410595381898829714373e-03
            7.37522761587901753976836971560260e-01 2.23633835367293892781859199203609e-01 3.99169299767275758544782249259697e-03
            7.37522761587902198066046821622876e-01 3.88434030448043601801977331433591e-02 3.99169299767285993413290512421554e-03
            2.23633835367292616025380880273588e-01 7.37522761587902642155256671685493e-01 3.99169299767286427094159506623328e-03
            2.23633835367291838869263642664009e-01 3.88434030448045336525453308240685e-02 3.99169299767285733204769115900490e-03
            3.88434030448042144634257510915631e-02 7.37522761587902753177559134201147e-01 3.99169299767286080149464311261909e-03
            3.88434030448049708028612769794563e-02 2.23633835367293809515132352316868e-01 3.99169299767291371056066040523547e-03
            6.39399415339909960032116487127496e-01 3.21417139704212873052568966159015e-01 4.49591014965592446900322087799395e-03
            6.39399415339909293898301712033572e-01 3.91834449558776665156756280339323e-02 4.49591014965595222457883650690746e-03
            3.21417139704212206918754191065091e-01 6.39399415339910071054418949643150e-01 4.49591014965599819475094989229547e-03
            3.21417139704212651007964041127707e-01 3.91834449558775346766914537965931e-02 4.49591014965604503228480126608702e-03
            3.91834449558775277377975498893647e-02 6.39399415339908627764486936939647e-01 4.49591014965601380726223368355932e-03
            3.91834449558780897882037663748633e-02 3.21417139704213483675232509995112e-01 4.49591014965599385794225995027773e-03
            5.33728924897748147060383416828699e-01 4.26438949417778823125502185575897e-01 4.76024570864742261472324358351216e-03
            5.33728924897748924216500654438278e-01 3.98321256844731755308863796471996e-02 4.76024570864738792025372404737027e-03
            4.26438949417778212502838641739800e-01 5.33728924897748591149593266891316e-01 4.76024570864744343140495530519729e-03
            4.26438949417779711303921885701129e-01 3.98321256844731616530985718327429e-02 4.76024570864744516612843128200439e-03
            3.98321256844731755308863796471996e-02 5.33728924897746814792753866640851e-01 4.76024570864741133902064973426604e-03
            3.98321256844733212476583616989956e-02 4.26438949417778212502838641739800e-01 4.76024570864738705289198605896672e-03
            7.52006738431794996557755439425819e-01 1.55247740375082515784654901835893e-01 5.30455340853805846351720632014803e-03
            7.52006738431800214605971177661559e-01 9.27455211931191847440913988975808e-02 5.30455340853802116696247281879550e-03
            1.55247740375080545138786192183034e-01 7.52006738431799548472156402567634e-01 5.30455340853800902389814098114584e-03
            1.55247740375080239827454420264985e-01 9.27455211931215439680187273552292e-02 5.30455340853811397466843757797506e-03
            9.27455211931191292329401676397538e-02 7.52006738431800103583668715145905e-01 5.30455340853801422806856891156713e-03
            9.27455211931186435103668941337673e-02 1.55247740375083848052284452023741e-01 5.30455340853810790313627165915022e-03
            6.62157408951895676274546076456318e-01 2.44250018940710467152399587575928e-01 6.25493845362825803824158299448754e-03
            6.62157408951895787296848538971972e-01 9.35925721073948280182008829797269e-02 6.25493845362831875356324218273585e-03
            2.44250018940708829573438265470031e-01 6.62157408951895787296848538971972e-01 6.25493845362833349871278798559615e-03
            2.44250018940709912040887274997658e-01 9.35925721073947169958984204640728e-02 6.25493845362842023488658682595087e-03
            9.35925721073943839289910329171107e-02 6.62157408951895787296848538971972e-01 6.25493845362833089662757402038551e-03
            9.35925721073961880414060487964889e-02 2.44250018940713631288019769272069e-01 6.25493845362842023488658682595087e-03
            5.60254721210861617031184778170427e-01 3.44727467648918373122057801083429e-01 7.00010259963591927739878784109351e-03
            5.60254721210862838276511865842622e-01 9.50178111402209257807527365002898e-02 7.00010259963586723569450853688068e-03
            3.44727467648916152676008550770348e-01 5.60254721210862838276511865842622e-01 7.00010259963593835935702358597155e-03
            3.44727467648918983744721344919526e-01 9.50178111402207314917234270978952e-02 7.00010259963600080940215875102695e-03
            9.50178111402206898583600036545249e-02 5.60254721210862061120394628233043e-01 7.00010259963588198084405433974098e-03
            9.50178111402216474257187428520410e-02 3.44727467648920482545804588880856e-01 7.00010259963587330722667445570551e-03
            5.73642212535126372330296362633817e-01 2.57088626651898688457720254518790e-01 8.22880168315748110330343223495220e-03
            5.73642212535122597572012637101579e-01 1.69269160812976965369003323758079e-01 8.22880168315740130602353730182585e-03
            2.57088626651898188857359173198347e-01 5.73642212535124262906549574836390e-01 8.22880168315753314500771153916503e-03
            2.57088626651899243569232567097060e-01 1.69269160812976410257491011179809e-01 8.22880168315753834917813946958631e-03
            1.69269160812976271479612933035241e-01 5.73642212535124151884247112320736e-01 8.22880168315749845053819200302314e-03
            1.69269160812978797236993955266371e-01 2.57088626651900686859164579800563e-01 8.22880168315749671581471602621605e-03
            4.68278059755925679663590699419728e-01 3.59525370113814635164573019210366e-01 9.20337543779727734749585010831652e-03
            4.68278059755924569440566074263188e-01 1.72196570130260739883709675268619e-01 9.20337543779716632519338759266248e-03
            3.59525370113814135564211937889922e-01 4.68278059755925291085532080614939e-01 9.20337543779727214332542217789523e-03
            3.59525370113814912720329175499501e-01 1.72196570130260406816802287721657e-01 9.20337543779723571413242666494625e-03
            1.72196570130260601105831597124052e-01 4.68278059755924958018624693067977e-01 9.20337543779719755021595517519017e-03
            1.72196570130261378261948834733630e-01 3.59525370113815190276085331788636e-01 9.20337543779717673353424345350504e-03
            ];
        
    case  25
        
        % ALG. DEG.:   25
        % PTS CARD.:  136
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.4e-16
        
        xyw=[
            3.33333333333332593184650249895640e-01 3.33333333333333481363069950020872e-01 9.56675462978373927769304430057673e-03
            5.71433885823402661074954522746339e-03 5.71433885823497897393785649455822e-03 2.01688465366838595022036528803255e-04
            5.71433885823444728119246960318378e-03 9.88571322283530284913410923763877e-01 2.01688465366858137766195580020678e-04
            9.88571322283531395136435548920417e-01 5.71433885823455309932450418841654e-03 2.01688465366824256448305408007116e-04
            2.72494286004120213340318201744594e-02 2.72494286004143874968530525393362e-02 1.02447302875085209661332186215077e-03
            2.72494286004084998453755872560578e-02 9.45501142799175964270830263558310e-01 1.02447302875087443117807506354211e-03
            9.45501142799175964270830263558310e-01 2.72494286004124376676660546081621e-02 1.02447302875080048858991155213971e-03
            7.66556657710398547944663505404606e-02 7.66556657710376621039927158562932e-02 3.02778006648536830713869250075732e-03
            7.66556657710362188140607031527907e-02 8.46688668457924675792014568287414e-01 3.02778006648529067826314253863984e-03
            8.46688668457927229304971206147457e-01 7.66556657710359690138801624925691e-02 3.02778006648533491371177994722075e-03
            4.82135684364817795888313867180841e-01 4.82135684364816408109533085735166e-01 3.86312189710397461614443948008102e-03
            4.82135684364817573843708942149533e-01 3.57286312703658168188347588056786e-02 3.86312189710415199161985810860642e-03
            3.57286312703654906908212751659448e-02 4.82135684364817795888313867180841e-01 3.86312189710421444166499327366182e-03
            1.49523257795367153777377211554267e-01 1.49523257795364461486542495549656e-01 5.57896389528127721602057675909236e-03
            1.49523257795361963484737088947440e-01 7.00953484409271410093822396447649e-01 5.57896389528128155282926670111010e-03
            7.00953484409271299071519933931995e-01 1.49523257795363379019093486022030e-01 5.57896389528124425627453319975757e-03
            4.22530364706221017723208888128283e-01 4.22530364706225070037248769949656e-01 7.82027325970425374324079825782974e-03
            4.22530364706225347593004926238791e-01 1.54939270587553634683786185632925e-01 7.82027325970429017243379377077872e-03
            1.54939270587552690994215254249866e-01 4.22530364706223404702711832214845e-01 7.82027325970421731404780274488076e-03
            2.35017663942219401373989740022807e-01 2.35017663942217097661213642822986e-01 8.11176381544534592926698479686820e-03
            2.35017663942217486239272261627775e-01 5.29964672115563639742674695298774e-01 8.11176381544522102917671446675740e-03
            5.29964672115565527121816558064893e-01 2.35017663942220955686224215241964e-01 8.11176381544532164313832112156888e-03
            9.68738481181608435122143418993801e-01 2.64387673203299901347751443836387e-02 3.27993024109075360927878506345223e-04
            9.68738481181608435122143418993801e-01 4.82275149806082274045460067668500e-03 3.27993024109104851226970112065828e-04
            2.64387673203289839951590778355239e-02 9.68738481181608435122143418993801e-01 3.27993024109130763658892515621801e-04
            2.64387673203308991298765562305562e-02 4.82275149806150448678065956187311e-03 3.27993024109142473042355359069688e-04
            4.82275149805882780845722734852643e-03 9.68738481181608435122143418993801e-01 3.27993024108947479281633841097232e-04
            4.82275149806148627218416180539862e-03 2.64387673203308991298765562305562e-02 3.27993024109069668866472957446945e-04
            9.35914465591766275842644517979352e-01 5.82420655518004273809573589915090e-02 5.90876590534632748809296121095258e-04
            9.35914465591761723928243554837536e-01 5.84346885643438184593234652197680e-03 5.90876590534703005110073181782582e-04
            5.82420655518003371753366081975400e-02 9.35914465591765942775737130432390e-01 5.90876590534699427242903979617950e-04
            5.82420655518037094777739071105316e-02 5.84346885643469496351976033565734e-03 5.90876590534723604951350406366828e-04
            5.84346885643483113931262451501425e-03 9.35914465591766275842644517979352e-01 5.90876590534645867655583195698910e-04
            5.84346885643377469271575463949375e-03 5.82420655518003371753366081975400e-02 5.90876590534657034937959796394580e-04
            8.85737420571855227358071260823635e-01 1.07957673880294022228376604743971e-01 9.65838800622361814789129663694212e-04
            8.85737420571854561224256485729711e-01 6.30490554785023606804150730908987e-03 9.65838800622276271237720557394368e-04
            1.07957673880294591217676725136698e-01 8.85737420571854339179651560698403e-01 9.65838800622400195546035650551175e-04
            1.07957673880294896529008497054747e-01 6.30490554785039566260129717534255e-03 9.65838800622306520478332902968077e-04
            6.30490554784987871500545608682842e-03 8.85737420571860889495496849121992e-01 9.65838800622277463860110291449246e-04
            6.30490554785099154011529520857948e-03 1.07957673880293189561108135876566e-01 9.65838800622393690333000737524571e-04
            8.16939712186552924322313629090786e-01 1.76404098155251098933860021134024e-01 1.30128537426810273608390744470853e-03
            8.16939712186550592853961916262051e-01 6.65618965819580327147875209448102e-03 1.30128537426810186872216945630498e-03
            1.76404098155251293222889330536418e-01 8.16939712186552813300011166575132e-01 1.30128537426809384562609306357217e-03
            1.76404098155253291624333655818191e-01 6.65618965819602878553062907940330e-03 1.30128537426814133368124792866638e-03
            6.65618965819592383476033248257409e-03 8.16939712186552924322313629090786e-01 1.30128537426816800505469107207546e-03
            6.65618965819600363204022741570043e-03 1.76404098155251293222889330536418e-01 1.30128537426813569582995100404332e-03
            7.34078265782891925539388466859236e-01 2.59206444374593891755154118072824e-01 1.51993199020561495921066708092440e-03
            7.34078265782887817714197353780037e-01 6.71528984251453138487608640616600e-03 1.51993199020570538167185237199419e-03
            2.59206444374594335844363968135440e-01 7.34078265782891259405573691765312e-01 1.51993199020566569987233940253191e-03
            2.59206444374597722024589074862888e-01 6.71528984251433969793199096898206e-03 1.51993199020568348078796816480462e-03
            6.71528984251449061887440095119928e-03 7.34078265782889372026431828999193e-01 1.51993199020568456499014065030906e-03
            6.71528984251436051461370269066720e-03 2.59206444374595501578539824549807e-01 1.51993199020568456499014065030906e-03
            6.41753644337056394775231638050172e-01 3.51505881363440575437806501213345e-01 1.64346540552525290437846372526565e-03
            6.41753644337053841262275000190130e-01 6.74047429950344525323435718178189e-03 1.64346540552529844086970811645187e-03
            3.51505881363440519926655269955518e-01 6.41753644337056172730626713018864e-01 1.64346540552531860703011634683435e-03
            3.51505881363444683262997614292544e-01 6.74047429950306881824007021464240e-03 1.64346540552525854222976064988870e-03
            6.74047429950314601343475118255810e-03 6.41753644337056394775231638050172e-01 1.64346540552521669202590270941755e-03
            6.74047429950329346493020921116113e-03 3.51505881363442962417309445299907e-01 1.64346540552532576276445475116361e-03
            5.45201712627926382559451212728163e-01 4.47959453755782277539765345863998e-01 1.70087181844667941973958669166223e-03
            5.45201712627927492782475837884704e-01 6.83883361629142316751028829457937e-03 1.70087181844665339888744703955581e-03
            4.47959453755781500383648108254420e-01 5.45201712627927159715568450337742e-01 1.70087181844670175430433989305357e-03
            4.47959453755785386164234296302311e-01 6.83883361629121500069317107772804e-03 1.70087181844663865373790123669551e-03
            6.83883361629120372499057722848192e-03 5.45201712627929824250827550713439e-01 1.70087181844662846223747987295383e-03
            6.83883361629130000214349394127566e-03 4.47959453755785386164234296302311e-01 1.70087181844669785117651894523760e-03
            8.99672974696369753822011716692941e-01 6.92285554477007208928895920507784e-02 1.79780310360282613232896231636460e-03
            8.99672974696373417557992979709525e-01 3.10984698559305557108434214796944e-02 1.79780310360263943271486031250106e-03
            6.92285554476950171221005803090520e-02 8.99672974696373861647202829772141e-01 1.79780310360285648998979191048875e-03
            6.92285554476976955351474884992058e-02 3.10984698559316555255271907753922e-02 1.79780310360273310778256306008416e-03
            3.10984698559315132782021606772105e-02 8.99672974696363092483863965753699e-01 1.79780310360280943561550603959631e-03
            3.10984698559315861365881517031085e-02 6.92285554477025388830924157446134e-02 1.79780310360286321204326132061624e-03
            8.36754745906223784679411892284406e-01 1.29450436723390405191480567737017e-01 2.60148497343646234111758097640177e-03
            8.36754745906235442021170456428081e-01 3.37948173703829651826069380149420e-02 2.60148497343622078087355120601387e-03
            1.29450436723384493253874438778439e-01 8.36754745906229335794535018067108e-01 2.60148497343648272411842370388513e-03
            1.29450436723384493253874438778439e-01 3.37948173703840337722681397281121e-02 2.60148497343631835906907490141293e-03
            3.37948173703825766045483192101528e-02 8.36754745906224228768621742347023e-01 2.60148497343624246491700091610255e-03
            3.37948173703861778904844470616808e-02 1.29450436723389350479607173838303e-01 2.60148497343633700734644165208920e-03
            7.60018820808681438983001044107368e-01 2.05092818400494147912382914000773e-01 3.19798198041868712154789378132591e-03
            7.60018820808689321566475882718805e-01 3.48883607908248641327197958617035e-02 3.19798198041883804249030376354312e-03
            2.05092818400489540486830719601130e-01 7.60018820808685990897402007249184e-01 3.19798198041881939421293701286686e-03
            2.05092818400489040886469638280687e-01 3.48883607908252943441418381098629e-02 3.19798198041882589942597192589346e-03
            3.48883607908255163887467631411710e-02 7.60018820808677997291624706122093e-01 3.19798198041872441810262728267844e-03
            3.48883607908248988271893153978453e-02 2.05092818400489096397620869538514e-01 3.19798198041860342114017790038361e-03
            6.72878720973754451684101240971358e-01 2.92117203365808597581576577795204e-01 3.57263564892233796005394097505814e-03
            6.72878720973758337464687429019250e-01 3.50040756604392336304165667115740e-02 3.57263564892263242936398803806242e-03
            2.92117203365807209802795796349528e-01 6.72878720973754229639496315940050e-01 3.57263564892261811789531122940389e-03
            2.92117203365804434245234233458177e-01 3.50040756604377209515455149357877e-02 3.57263564892242946671729875163237e-03
            3.50040756604382483074822118851444e-02 6.72878720973754451684101240971358e-01 3.57263564892227160688098486218678e-03
            3.50040756604383454519968665863416e-02 2.92117203365801103576160357988556e-01 3.57263564892244074241989260087848e-03
            5.78766335729510972285538628057111e-01 3.85847444474737621966653478011722e-01 3.79149504573928278206018660512200e-03
            5.78766335729513636820797728432808e-01 3.53862197957528004654825792840711e-02 3.79149504573951393396336051466733e-03
            3.85847444474738177078165790589992e-01 5.78766335729509084906396765290992e-01 3.79149504573943413668346558154099e-03
            3.85847444474734901920243146378198e-01 3.53862197957514057478078939311672e-02 3.79149504573938036025571030052106e-03
            3.53862197957515028923225486323645e-02 5.78766335729514747043822353589348e-01 3.79149504573929796089060140218407e-03
            3.53862197957522592317580745202577e-02 3.85847444474732015340379120971193e-01 3.79149504573953865377289318416842e-03
            7.76386479467535695064839273982216e-01 1.41173270259467076304105148665258e-01 4.22134752481821608899315023677445e-03
            7.76386479467535695064839273982216e-01 8.24402502729935232617108908925729e-02 4.22134752481836007104165631176329e-03
            1.41173270259466576703744067344815e-01 7.76386479467535695064839273982216e-01 4.22134752481817792507667874701838e-03
            1.41173270259473793153404130862327e-01 8.24402502729952163518234442562971e-02 4.22134752481835053006253843932427e-03
            8.24402502729926489610789985817974e-02 7.76386479467535695064839273982216e-01 4.22134752481824904873919379610925e-03
            8.24402502729979641538093915187346e-02 1.41173270259467881215798001903750e-01 4.22134752481816838409756087457936e-03
            6.95281183841358396691134657885414e-01 2.20694701488690830926131525302480e-01 5.02216039340301970139712750551553e-03
            6.95281183841349070817727806570474e-01 8.40241146699540614184442688383569e-02 5.02216039340325475642812236287682e-03
            2.20694701488689193347170203196583e-01 6.95281183841358507713437120401068e-01 5.02216039340308648825095261258866e-03
            2.20694701488698713509606363913917e-01 8.40241146699534091624173015588894e-02 5.02216039340306480420750290249998e-03
            8.40241146699542557074735782407515e-02 6.95281183841349070817727806570474e-01 5.02216039340313332578480398638021e-03
            8.40241146699528540509049889806192e-02 2.20694701488689720703106900145940e-01 5.02216039340303965071710123879711e-03
            6.04556578856266901844662697840249e-01 3.10779394840663303334338252170710e-01 5.60344952507376564687646691709233e-03
            6.04556578856259130283490321744466e-01 8.46640263030742495908853584296594e-02 5.60344952507366503291486026228085e-03
            3.10779394840659584087205757896299e-01 6.04556578856267012866965160355903e-01 5.60344952507376738159994289389942e-03
            3.10779394840667078092621977702947e-01 8.46640263030706413660553266709030e-02 5.60344952507344992720383913820115e-03
            8.46640263030713630110213330226543e-02 6.04556578856262349930261734698433e-01 5.60344952507357222520889550310130e-03
            8.46640263030726120119240363237623e-02 3.10779394840659306531449601607164e-01 5.60344952507371447253392560128304e-03
            5.06851226677805466458437422261341e-01 4.07374461989679470530489879820379e-01 5.94839939536055337110376228793029e-03
            5.06851226677805355436134959745687e-01 8.57743113325167561011852512820042e-02 5.94839939536034260220143110586832e-03
            4.07374461989673308792703210201580e-01 5.06851226677810018372838385403156e-01 5.94839939536050219676122097212101e-03
            4.07374461989678027240557867116877e-01 8.57743113325149242331946197737125e-02 5.94839939536028622368846185963775e-03
            8.57743113325147021885896947424044e-02 5.06851226677805466458437422261341e-01 5.94839939536044321616303776067980e-03
            8.57743113325157430226752808266610e-02 4.07374461989673586348459366490715e-01 5.94839939536050740093164890254229e-03
            6.17605377498659180446338723413646e-01 2.30766928655942393788791378028691e-01 6.61764978454050002698583199389759e-03
            6.17605377498654406487332835240522e-01 1.51627693845405031591866418239078e-01 6.61764978454038640259815551303291e-03
            2.30766928655939534964502968250599e-01 6.17605377498657959201011635741452e-01 6.61764978454055033396663532130333e-03
            2.30766928655944475456962550197204e-01 1.51627693845402283789880470976641e-01 6.61764978454038987204510746664710e-03
            1.51627693845403532790783174277749e-01 6.17605377498652519108190972474404e-01 6.61764978454035951438427787252294e-03
            1.51627693845403033190422092957306e-01 2.30766928655941672143825371676940e-01 6.61764978454052257839101969238982e-03
            5.23270463532956031649234773794888e-01 3.23002638196493885303084425686393e-01 7.48823901554941835645440306734599e-03
            5.23270463532956031649234773794888e-01 1.53726898270555273340320923125546e-01 7.48823901554930126261977463286712e-03
            3.23002638196489277877532231286750e-01 5.23270463532956031649234773794888e-01 7.48823901554935243696231594867641e-03
            3.23002638196494051836538119459874e-01 1.53726898270550610403617497468076e-01 7.48823901554933508972755618060546e-03
            1.53726898270550860203798038128298e-01 5.23270463532956031649234773794888e-01 7.48823901554922666951030763016206e-03
            1.53726898270553441472330291617254e-01 3.23002638196491276278976556568523e-01 7.48823901554934029389798411102674e-03
            4.31308082067836551232176134362817e-01 3.30276680517266163494838338010595e-01 9.01193261972355053046612738398835e-03
            4.31308082067838161055561840839800e-01 2.38415237414900393897454478064901e-01 9.01193261972365981804511392283530e-03
            3.30276680517262999359218156314455e-01 4.31308082067836440209873671847163e-01 9.01193261972349154986794417254714e-03
            3.30276680517265996961384644237114e-01 2.38415237414897340784136758884415e-01 9.01193261972367369583292173729205e-03
            2.38415237414896674650321983790491e-01 4.31308082067837994522108147066319e-01 9.01193261972356961242436312886639e-03
            2.38415237414898950607522465361399e-01 3.30276680517264054071091550213168e-01 9.01193261972360604161735864181537e-03
            ];
        
    case  27
        
        % ALG. DEG.:   27
        % PTS CARD.:  153
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.0e-16
        
        xyw=[
            6.02872510688659112715326315878883e-03 6.02872510688349898255733450014304e-03 2.21617807147030706615281614446644e-04
            6.02872510688389449950985721216057e-03 9.87942549786234525122097238636343e-01 2.21617807146858643730508164892967e-04
            9.87942549786236745568146488949424e-01 6.02872510688300979053710904054242e-03 2.21617807146739815172403753607000e-04
            2.25804480641049951095133963008266e-02 2.25804480640832486160185510470910e-02 8.54194011283785389725131675930925e-04
            2.25804480641088878289934882559464e-02 9.54839103871812144852526671456872e-01 8.54194011283423591460173263101296e-04
            9.54839103871770289444498303055298e-01 2.25804480641217698855260920254295e-02 8.54194011284169305714408793051007e-04
            4.96948162575856389189254969096510e-01 4.96948162575845342470159948788933e-01 1.46829707080454053419460436202826e-03
            4.96948162575851670741400312181213e-01 6.10367484829858492761944788185247e-03 1.46829707080504035139612017957234e-03
            6.10367484829798297857328392979070e-03 4.96948162575852780964424937337753e-01 1.46829707080488596100675824374093e-03
            6.97453864826078351324412096801098e-02 6.97453864825942071448139358835761e-02 2.26420768932116987445790634581044e-03
            6.97453864825669234139837726615951e-02 8.60509227034811585710372128232848e-01 2.26420768932207713483584221592082e-03
            8.60509227034811585710372128232848e-01 6.97453864825740427191291814779106e-02 2.26420768932204677717501262179667e-03
            1.33222990598758694869019336692872e-01 1.33222990598769824854841203887190e-01 4.63019691359635183341980635418622e-03
            1.33222990598766410919040481530828e-01 7.33554018802442087121562508400530e-01 4.63019691359698674221201386558278e-03
            7.33554018802459406600746660842560e-01 1.33222990598787810467840131423145e-01 4.63019691359614713604964109094908e-03
            4.61535910661840520496923545579193e-01 4.61535910661833803647624563382124e-01 5.16250885710355187502296203660990e-03
            4.61535910661845460989383127525798e-01 7.69281786763297420472795806745125e-02 5.16250885710394565725200877182033e-03
            7.69281786763209435298094263089297e-02 4.61535910661838633117781682813074e-01 5.16250885710316763377303317383848e-03
            2.11399010110189911060274425835814e-01 2.11399010110171869936124267042032e-01 6.57682721325546226776026514926343e-03
            2.11399010110194796041582776524592e-01 5.77201979779594975816792157274904e-01 6.57682721325511185361811783423036e-03
            5.77201979779581431095891730365111e-01 2.11399010110200957779369446143392e-01 6.57682721325748148588630215272133e-03
            3.92179397457324996345562340138713e-01 3.92179397457366463175532089735498e-01 7.93730408209934214236458416280584e-03
            3.92179397457315004338340713729849e-01 2.15641205085320392109693443671858e-01 7.93730408209899693239286477819405e-03
            2.15641205085286835618774148315424e-01 3.92179397457343759114678505284246e-01 7.93730408209961622867378849832676e-03
            3.02074000493948113810205313711776e-01 3.02074000493992356197736626199912e-01 8.44480156880698766552750811342776e-03
            3.02074000493948557899415163774393e-01 3.95851999012071742445328936810256e-01 8.44480156880705011557264327848316e-03
            3.95851999012003130662407102136058e-01 3.02074000494003291894529183991835e-01 8.44480156880698593080403213662066e-03
            9.72577577219754130233297928498359e-01 2.56411289038486409075368044341303e-02 1.50856366727572321169079105906974e-04
            9.72577577219754130233297928498359e-01 1.78129387641991015177322932316883e-03 1.50856366728038907574008242740149e-04
            2.56411289038599408962593173555433e-02 9.72577577219763789173612167360261e-01 1.50856366727007614467540030922521e-04
            2.56411289038599408962593173555433e-02 1.78129387637662663264326301515439e-03 1.50856366726891767465409954773747e-04
            1.78129387640009765811366371224267e-03 9.72577577219754130233297928498359e-01 1.50856366727610349560279034974997e-04
            1.78129387638326411834321927329938e-03 2.56411289038599408962593173555433e-02 1.50856366727299590112590377266599e-04
            9.44757428656674114186841961782193e-01 4.89478566028413442090538865159033e-02 5.17021387795440720037964954514109e-04
            9.44757428656674114186841961782193e-01 6.29471474046813285474488708359786e-03 5.17021387795043359941749017139045e-04
            4.89478566028448414115814557590056e-02 9.44757428656667563870996673358604e-01 5.17021387795696049649585290808318e-04
            4.89478566028448414115814557590056e-02 6.29471474048211993013168807920010e-03 5.17021387795682388702211973452449e-04
            6.29471474048273402224218386891152e-03 9.44757428656674114186841961782193e-01 5.17021387795627419652066958377645e-04
            6.29471474049513469301020407442593e-03 4.89478566028448414115814557590056e-02 5.17021387796343209926341888404977e-04
            9.00140112535737002552593821746996e-01 9.43702253266477247839461028888763e-02 7.96599751112111792092362883010992e-04
            9.00140112535686154338065989577444e-01 5.48966213761783138058714115459225e-03 7.96599751112452556835175077054600e-04
            9.43702253266545387777597397871432e-02 9.00140112535729897125236220745137e-01 7.96599751112124910938649957614643e-04
            9.43702253266842372436684627245995e-02 5.48966213761628400724656984266403e-03 7.96599751112486058682304879141611e-04
            5.48966213761158811079710062585946e-03 9.00140112535713687869076693459647e-01 7.96599751111877279162454268401916e-04
            5.48966213761193679021577196408543e-03 9.43702253266681945209626292125904e-02 7.96599751111992963534258471725025e-04
            8.35926640580037294547821602463955e-01 1.58178278042184511376078148714441e-01 1.05401251886926071270333959972731e-03
            8.35926640580002766611755760095548e-01 5.89508137778006844481604176166911e-03 1.05401251886919002272169354483822e-03
            1.58178278042176712059330156989745e-01 8.35926640580041735439920103090117e-01 1.05401251886950162242606587881255e-03
            1.58178278042216791110519125140854e-01 5.89508137777945955687597390237897e-03 1.05401251886914817251783560436706e-03
            5.89508137778011354762641715865357e-03 8.35926640580004987057805010408629e-01 1.05401251886940144214532821820285e-03
            5.89508137777934333040308345630365e-03 1.58178278042213321663567171526665e-01 1.05401251886928608303417576053107e-03
            7.61488498658509094241253478685394e-01 2.32483150304506530625658911048959e-01 1.21041899004435812500779512390636e-03
            7.61488498658509094241253478685394e-01 6.02835103698335251359852193786537e-03 1.21041899004361804860485651857971e-03
            2.32483150304480662429185144901567e-01 7.61488498658535295504634632379748e-01 1.21041899004415624656327832298075e-03
            2.32483150304507835137712845607894e-01 6.02835103698241836500670842724503e-03 1.21041899004340836390469782202217e-03
            6.02835103698410798567230983735499e-03 7.61488498658503099036920502840076e-01 1.21041899004381038607025544706630e-03
            6.02835103698397788141161157682291e-03 2.32483150304507807382137229978980e-01 1.21041899004369372591649600678920e-03
            6.79274857925002639724709752044873e-01 3.14735969095647993842845835388289e-01 1.31787037168714532033020336854179e-03
            6.79274857925028618943485980707919e-01 5.98917297934907846834740041686018e-03 1.31787037168685453730754275625259e-03
            3.14735969095626788583075494898367e-01 6.79274857925025066229807180206990e-01 1.31787037168712125104197419034335e-03
            3.14735969095621681557162219178281e-01 5.98917297934821457605636396692717e-03 1.31787037168668236600255205814847e-03
            5.98917297934755538113549278023129e-03 6.79274857925024622140597330144374e-01 1.31787037168649198010106360356986e-03
            5.98917297934768461803445305235982e-03 3.14735969095620127244927743959124e-01 1.31787037168655876695488871064299e-03
            5.90173779527676689582449398585595e-01 4.03765902224775929951761099800933e-01 1.42354344170825546263270400260126e-03
            5.90173779527698449953732051653787e-01 6.06031824754737352689559770624328e-03 1.42354344170857183282663527279510e-03
            4.03765902224775929951761099800933e-01 5.90173779527677799805474023742136e-01 1.42354344170834805349823426467992e-03
            4.03765902224754891225444453084492e-01 6.06031824754691989670662977118809e-03 1.42354344170850569649411365702463e-03
            6.06031824754597100296527045770745e-03 5.90173779527697783819917276559863e-01 1.42354344170818802525757540422546e-03
            6.06031824754629279417006415542346e-03 4.03765902224754891225444453084492e-01 1.42354344170839207210643717615994e-03
            9.08147639531788897215847100596875e-01 6.24216082470002639470685323885846e-02 1.52265358333221012369917968243271e-03
            9.08147639531850403571411334269214e-01 2.94307522211805228096181963337585e-02 1.52265358333129961071472635580903e-03
            6.24216082469753116845900819953386e-02 9.08147639531814876434623329259921e-01 1.52265358333123564278654971104743e-03
            6.24216082469878572047683462642453e-02 2.94307522211964094072111919331292e-02 1.52265358333170119919941498665139e-03
            2.94307522211861953553846404929573e-02 9.08147639531878825280841738276649e-01 1.52265358333020586756312297893601e-03
            2.94307522212155676932798797906798e-02 6.24216082469528851794926538332220e-02 1.52265358333144424328453592210053e-03
            8.53441207153422753961535818234552e-01 1.17038555323248083528397955888067e-01 2.05571787626365872977962645506977e-03
            8.53441207153414649333456054591807e-01 2.95202375233448027769256327701441e-02 2.05571787626518615380022403371640e-03
            1.17038555323194598534186638971732e-01 8.53441207153465941637193736823974e-01 2.05571787626350260466678854243128e-03
            1.17038555323250345607810629644518e-01 2.95202375233389845143872065591495e-02 2.05571787626461022560619973376106e-03
            2.95202375233244475316585209156983e-02 8.53441207153490366543735490267863e-01 2.05571787626424549999537561006946e-03
            2.95202375233212903349322431267865e-02 1.17038555323194598534186638971732e-01 2.05571787626370600099434682306310e-03
            7.84561667827341380210270926909288e-01 1.84167433829063698080830135950237e-01 2.51430251702616921680166761632336e-03
            7.84561667827341380210270926909288e-01 3.12708983435956086593954239560844e-02 2.51430251702615837477994276127902e-03
            1.84167433828987842092672622129612e-01 7.84561667827407882569445973786060e-01 2.51430251702621822273986396112377e-03
            1.84167433829062893169137282711745e-01 3.12708983435923959515179149093456e-02 2.51430251702540637215310681540359e-03
            3.12708983436014859025320333785203e-02 7.84561667827366804317534842994064e-01 2.51430251702771095229094200362852e-03
            3.12708983435994736232999002822908e-02 1.84167433829027227254471199557884e-01 2.51430251702702530283706217062445e-03
            7.08519881821619734729722495103488e-01 2.60168754876507091200465993097168e-01 2.81439378294814305375703966660694e-03
            7.08519881821622954376493908057455e-01 3.13113633018684209274873353479052e-02 2.81439378294799690330418862060924e-03
            2.60168754876437591239124458297738e-01 7.08519881821694563761582230654312e-01 2.81439378294881612646571866775957e-03
            2.60168754876498486972025148133980e-01 3.13113633018632098181655010193936e-02 2.81439378294745480221794586839223e-03
            3.13113633018666237539662233757554e-02 7.08519881821631503093783521762816e-01 2.81439378294824843820820525763793e-03
            3.13113633018663253815283553649351e-02 2.60168754876487051674871509021614e-01 2.81439378294819162601436701720559e-03
            6.23662680919337741869412639061920e-01 3.44939109327516923997336562024429e-01 3.13542669889337304728327282532518e-03
            6.23662680919329526219030412903521e-01 3.13982097531447859606323902426084e-02 3.13542669889353654497088363939383e-03
            3.44939109327494275447634208831005e-01 6.23662680919366607668052893131971e-01 3.13542669889453531201217728607844e-03
            3.44939109327520476711015362525359e-01 3.13982097531416357028000163609249e-02 3.13542669889358424986647300158893e-03
            3.13982097531362372433427765372471e-02 6.23662680919338519025529876671499e-01 3.13542669889271732180935359224350e-03
            3.13982097531376180832296540756943e-02 3.44939109327520476711015362525359e-01 3.13542669889350228418223309745372e-03
            8.02460515514437200401687277917517e-01 1.25010019659880150522113240185718e-01 3.26908364108432960420946855606417e-03
            8.02460515514371586220931931165978e-01 7.25294648257160112780894678508048e-02 3.26908364108405768630460919155212e-03
            1.25010019659864191066134253560449e-01 8.02460515514425098970718863711227e-01 3.26908364108519262913876701759364e-03
            1.25010019659891807863871804329392e-01 7.25294648257028690130354675602575e-02 3.26908364108346701296103908873647e-03
            7.25294648256896296034668125685130e-02 8.02460515514371586220931931165978e-01 3.26908364108515706730750949304820e-03
            7.25294648256799429075769580776978e-02 1.25010019659892085419627960618527e-01 3.26908364108442978449020621667387e-03
            5.31386343848361364372578918846557e-01 4.36826791108310630296074350553681e-01 3.35237647732895306662559420374237e-03
            5.31386343848350817253844979859423e-01 3.17868650433291086154774518490740e-02 3.35237647732893138258214449365369e-03
            4.36826791108323175816252614822588e-01 5.31386343848351594409962217469001e-01 3.35237647732938718117545739971774e-03
            4.36826791108322454171286608470837e-01 3.17868650433278804312564602696511e-02 3.35237647732899079686119669929667e-03
            3.17868650433239946506702722217597e-02 5.31386343848359476993437056080438e-01 3.35237647732833073457858752419725e-03
            3.17868650433250701792253778421582e-02 4.36826791108329004487131896894425e-01 3.35237647732881298770490907656949e-03
            7.29210554614469153023037506500259e-01 1.95225741421010040577144195594883e-01 4.05199187592837108129462109218366e-03
            7.29210554614469153023037506500259e-01 7.55637039645158242739952925148827e-02 4.05199187592683845310359558311575e-03
            1.95225741420998188946356322048814e-01 7.29210554614469153023037506500259e-01 4.05199187592896262199992918340286e-03
            1.95225741421000270614527494217327e-01 7.55637039645057212444712035903649e-02 4.05199187592689916842525477136405e-03
            7.55637039645300906398617257764272e-02 7.29210554614403982931492009811336e-01 4.05199187592761387449735721588695e-03
            7.55637039645257468922778798514628e-02 1.95225741421016812937594409049780e-01 4.05199187592777086697193311692899e-03
            6.47395307940577979799456898035714e-01 2.77031342028366378560377825124306e-01 4.60801434777555109917335229852142e-03
            6.47395307940592967810289337649010e-01 7.55733500310482725348393273634429e-02 4.60801434777506017242965086211370e-03
            2.77031342028364047092026112295571e-01 6.47395307940594855189431200415129e-01 4.60801434777543227061524788723545e-03
            2.77031342028360938467557161857258e-01 7.55733500310375727604395024172845e-02 4.60801434777551987415078471599372e-03
            7.55733500310353661921780599186604e-02 6.47395307940563879967044158547651e-01 4.60801434777320662039556964373332e-03
            7.55733500310362127372343366005225e-02 2.77031342028365767937714281288208e-01 4.60801434777436541567752215087239e-03
            5.56614640862459930659156270849053e-01 3.67086174365095052607443903980311e-01 5.01105544489713772976502426104162e-03
            5.56614640862468257331840959523106e-01 7.62991847724459326673951409247820e-02 5.01105544489729819168655211569785e-03
            3.67086174365103157235523667623056e-01 5.56614640862462817239020296256058e-01 5.01105544489699374771651818605278e-03
            3.67086174365103212746674898880883e-01 7.62991847724408395192696730191528e-02 5.01105544489763472804089161627417e-03
            7.62991847724286409437866041116649e-02 5.56614640862459930659156270849053e-01 5.01105544489563806132004231130850e-03
            7.62991847724316246681652842198673e-02 3.67086174365097550609249310582527e-01 5.01105544489678124409071102718372e-03
            6.55804993103131339537981148168910e-01 2.07677817553296312214072827373457e-01 5.44629424133547934000487700245685e-03
            6.55804993103188182956841956183780e-01 1.36517189343557499014991662988905e-01 5.44629424133495111670644206469660e-03
            2.07677817553273053041706930343935e-01 6.55804993103150879463214550924022e-01 5.44629424133435610655418201986322e-03
            2.07677817553268279082701042170811e-01 1.36517189343538458690119341554237e-01 5.44629424133436217808634793868805e-03
            1.36517189343568823289842839585617e-01 6.55804993103131450560283610684564e-01 5.44629424133391808887649787607188e-03
            1.36517189343564021575261335783580e-01 2.07677817553268306838276657799724e-01 5.44629424133334563012942552973072e-03
            5.70923789561694650096512759773759e-01 2.91806857017915921570505588533706e-01 6.13438674052210319159739171368528e-03
            5.70923789561736949593750978237949e-01 1.37269353420383682928829216507438e-01 6.13438674052282830601035001905075e-03
            2.91806857017880116877961427235277e-01 5.70923789561755157251354830805212e-01 6.13438674052058964536460194949541e-03
            2.91806857017890386440939209933276e-01 1.37269353420368556140118698749575e-01 6.13438674052204681308442246745471e-03
            1.37269353420353928951769262312155e-01 5.70923789561736949593750978237949e-01 6.13438674052050204182906512073714e-03
            1.37269353420356593487028362687852e-01 2.91806857017874565762838301452575e-01 6.13438674052064949332452314934017e-03
            4.78028270406235411549289437971311e-01 3.83097780854575353792768055427587e-01 6.59148576656336162432614500517047e-03
            4.78028270406262334457636598017416e-01 1.38873948739193980861372779145313e-01 6.59148576656389331707153189654491e-03
            3.83097780854526059890474698477192e-01 4.78028270406294142347292108752299e-01 6.59148576656286896285896759195566e-03
            3.83097780854547487194849963998422e-01 1.38873948739189817525030434808286e-01 6.59148576656360535297451974656724e-03
            1.38873948739170527399977572713397e-01 4.78028270406264887970593235877459e-01 6.59148576656320289712809312732134e-03
            1.38873948739172498045846282366256e-01 3.83097780854534442074310618409072e-01 6.59148576656340846185999637896202e-03
            4.88026817278799873722761049066321e-01 2.97907118579997409746340508718276e-01 7.57584990478472278946364681928571e-03
            4.88026817278788882514817260016571e-01 2.14066064141203715731620604856289e-01 7.57584990478452156154043350966276e-03
            2.97907118579970653371447042445652e-01 4.88026817278850555403835187462391e-01 7.57584990478376955891359756378733e-03
            2.97907118579965990434743616788182e-01 2.14066064141187561986612308828626e-01 7.57584990478385282564044445052787e-03
            2.14066064141161527256684848907753e-01 4.88026817278841729130789417467895e-01 7.57584990478407140079841752822176e-03
            2.14066064141163109324494939755823e-01 2.97907118579965990434743616788182e-01 7.57584990478445564204834639099317e-03
            ];
        
    case  28
        
        % ALG. DEG.:   28
        % PTS CARD.:  171
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.3e-16
        
        xyw=[
            3.52781468614848242593140525968920e-03 3.52781468614803963776416217967835e-03 8.44142358948589801095896412164166e-05
            3.52781468614818492085527523727251e-03 9.92944370627703354337256769213127e-01 8.44142358948607961482285544363435e-05
            9.92944370627703354337256769213127e-01 3.52781468614843515471668489169588e-03 8.44142358948624088989601266241891e-05
            2.32544785996821011531121570214964e-02 2.32544785996821011531121570214964e-02 6.49518356010246643642003228080739e-04
            2.32544785996818652307194241757315e-02 9.53491042800635790754881782049779e-01 6.49518356010244258397223759970984e-04
            9.53491042800635790754881782049779e-01 2.32544785996819727835749347377714e-02 6.49518356010250872030475921548032e-04
            6.13061711490497721066539327239298e-02 6.13061711490494321008526412697393e-02 1.96921130402785910309959582775718e-03
            6.13061711490498553733807796106703e-02 8.77387657701900081086421323561808e-01 1.96921130402786994512132068280152e-03
            8.77387657701900636197933636140078e-01 6.13061711490494321008526412697393e-02 1.96921130402786170518480979296783e-03
            4.85597365316512985522479084465886e-01 4.85597365316513485122840165786329e-01 2.89525271680968416046497715399255e-03
            4.85597365316512985522479084465886e-01 2.88052693669737236437100591501803e-02 2.89525271680971104867885479450251e-03
            2.88052693669737791548612904080073e-02 4.85597365316512985522479084465886e-01 2.89525271680970584450842686408123e-03
            1.19574379687203399824646510296589e-01 1.19574379687203594113675819698983e-01 3.76448424794176891156061159904311e-03
            1.19574379687202816957558582089405e-01 7.60851240625593949751248601387488e-01 3.76448424794175113064498283677040e-03
            7.60851240625593283617433826293563e-01 1.19574379687203302680131855595391e-01 3.76448424794174289070847194693670e-03
            1.93832849735945389468838584434707e-01 1.93832849735944889868477503114264e-01 5.47600158879724911026309897010833e-03
            1.93832849735944195979087112391426e-01 6.12334300528110331285347456287127e-01 5.47600158879727946792392856423248e-03
            6.12334300528110886396859768865397e-01 1.93832849735944889868477503114264e-01 5.47600158879725171234831293531897e-03
            4.36854825801193724643667337659281e-01 4.36854825801192891976398868791875e-01 5.65722704264194110879149235415753e-03
            4.36854825801192891976398868791875e-01 1.26290348397614132780475415529509e-01 5.65722704264195585394103815701783e-03
            1.26290348397614382580655956189730e-01 4.36854825801192558909491481244913e-01 5.65722704264192636364194655129722e-03
            2.74786691854845666060924713747227e-01 2.74786691854845777083227176262881e-01 7.11814525498565865702094157541069e-03
            2.74786691854845999127832101294189e-01 4.50426616290308667878150572505547e-01 7.11814525498561355421056617842623e-03
            4.50426616290308556855848109989893e-01 2.74786691854846387705890720098978e-01 7.11814525498563610561575387691846e-03
            3.61886974590005627838706914189970e-01 3.61886974590005849883311839221278e-01 7.45528516348590089557468729708489e-03
            3.61886974590005627838706914189970e-01 2.76226050819989354945249715456157e-01 7.45528516348593125323551689120904e-03
            2.76226050819989632501005871745292e-01 3.61886974590005461305253220416489e-01 7.45528516348590436502163925069908e-03
            9.76405945543714515189037683740025e-01 1.90656116400245469189034253076898e-02 2.44787511408620691698984073170209e-04
            9.76405945543714515189037683740025e-01 4.52844281626076268498781729476832e-03 2.44787511408625082717782639463167e-04
            1.90656116400246301856302721944303e-02 9.76405945543714737233642608771333e-01 2.44787511408620529068658200344544e-04
            1.90656116400246301856302721944303e-02 4.52844281626113478317341431989007e-03 2.44787511408633214234076280746422e-04
            4.52844281626104717963787749113180e-03 9.76405945543714515189037683740025e-01 2.44787511408624811667239518087058e-04
            4.52844281626053630357420232144250e-03 1.90656116400246301856302721944303e-02 2.44787511408623890095392905408289e-04
            9.49626039031084334673948887939332e-01 4.57527920669594106994892968032218e-02 3.85232149329054818828049766565869e-04
            9.49626039031083335473226725298446e-01 4.62116890195622860570967560533973e-03 3.85232149329059155636739708583605e-04
            4.57527920669597021330332609068137e-02 9.49626039031083668540134112845408e-01 3.85232149329069184506835199499619e-04
            4.57527920669602294889699578561704e-02 4.62116890195593717216571150174786e-03 3.85232149329042513133392056090543e-04
            4.62116890195595972357089920024009e-03 9.49626039031084334673948887939332e-01 3.85232149329043814175999038695863e-04
            4.62116890195662672474741228256789e-03 4.57527920669607637838005587127554e-02 3.85232149329081164940841164323615e-04
            9.09575444421073742162775488395710e-01 8.55585301859531771606626193715783e-02 5.96025956156551422852885036007820e-04
            9.09575444421073298073565638333093e-01 4.86602539297293842923686213453038e-03 5.96025956156533858777690770835989e-04
            8.55585301859525665379990755354811e-02 9.09575444421074630341195188520942e-01 5.96025956156532015633997545478451e-04
            8.55585301859539404389920491666999e-02 4.86602539297307807447667826750148e-03 5.96025956156547736565498585292744e-04
            4.86602539297305639043322855741280e-03 9.09575444421073298073565638333093e-01 5.96025956156551422852885036007820e-04
            4.86602539297277363050664433785641e-03 8.55585301859539404389920491666999e-02 5.96025956156519547309013962177460e-04
            8.53928600117039415806630131555721e-01 1.40745793360055215792669969232520e-01 8.46151182008891421422458467560546e-04
            8.53928600117039526828932594071375e-01 5.32560652290524263324789089324440e-03 8.46151182008873206825960711086054e-04
            1.40745793360055188037094353603607e-01 8.53928600117039526828932594071375e-01 8.46151182008888277236158259597687e-04
            1.40745793360055049259216275459039e-01 5.32560652290533457359211766402041e-03 8.46151182008881121501819855268423e-04
            5.32560652290529814439912215107142e-03 8.53928600117039415806630131555721e-01 8.46151182008881121501819855268423e-04
            5.32560652290531115482519197712463e-03 1.40745793360055854170909128697531e-01 8.46151182008885458310509797286159e-04
            7.86232363633783681855504710256355e-01 2.08266834439758047325952361461532e-01 1.01879942321813320160739380071391e-03
            7.86232363633783681855504710256355e-01 5.50080192645838010612191482096023e-03 1.01879942321813927313955971953874e-03
            2.08266834439758491415162211524148e-01 7.86232363633783126743992397678085e-01 1.01879942321814989832085007748219e-03
            2.08266834439757991814801130203705e-01 5.50080192645833153386458747036158e-03 1.01879942321814057418216670214406e-03
            5.50080192645825086922295454883169e-03 7.86232363633783681855504710256355e-01 1.01879942321812799743696587029262e-03
            5.50080192645841219850622039189147e-03 2.08266834439757991814801130203705e-01 1.01879942321813948997999421663962e-03
            7.10218209945296274909765088523272e-01 2.84299186848130214233520973721170e-01 1.11363014580781764448214055107655e-03
            7.10218209945296607976672476070235e-01 5.48260320657365136931549187693236e-03 1.11363014580783412435516233074395e-03
            2.84299186848130214233520973721170e-01 7.10218209945296163887462626007618e-01 1.11363014580782523389734794960759e-03
            2.84299186848130491789277130010305e-01 5.48260320657360106233468854952662e-03 1.11363014580782566757821694380937e-03
            5.48260320657359759288773659591243e-03 7.10218209945295941842857700976310e-01 1.11363014580782154760996149889252e-03
            5.48260320657359152135557067708760e-03 2.84299186848129603610857429885073e-01 1.11363014580781807816300954527833e-03
            9.19615743255533413247349017183296e-01 5.56921540915342391797615562154533e-02 1.11405868600329086698375036235120e-03
            9.19615743255533413247349017183296e-01 2.46921026529322192033522043175253e-02 1.11405868600328154284506698701307e-03
            5.56921540915337326405065709877817e-02 9.19615743255533413247349017183296e-01 1.11405868600328631333462592323258e-03
            5.56921540915340518296261507202871e-02 2.46921026529317196029911229970821e-02 1.11405868600327568815333556528913e-03
            2.46921026529319034836795765386341e-02 9.19615743255533413247349017183296e-01 1.11405868600329173434548835075475e-03
            2.46921026529327673759706129885672e-02 5.56921540915341767297164210503979e-02 1.11405868600328110916419799281130e-03
            6.28121500758708983802591774292523e-01 3.66399643439952127366154854826164e-01 1.18799277964062475684836606149020e-03
            6.28121500758709094824894236808177e-01 5.47885580133890270904117869577021e-03 1.18799277964064622405138127447799e-03
            3.66399643439951738788096236021374e-01 6.28121500758709427891801624355139e-01 1.18799277964063061154009748321414e-03
            3.66399643439952127366154854826164e-01 5.47885580133884372844299548432900e-03 1.18799277964062172108228310207778e-03
            5.47885580133884546316647146113610e-03 6.28121500758709094824894236808177e-01 1.18799277964062605789097304409552e-03
            5.47885580133881163605868991339776e-03 3.66399643439952127366154854826164e-01 1.18799277964064383880660180636823e-03
            5.41474950946130073603512755653355e-01 4.53014671446369310281454545474844e-01 1.24371131542504935289794598674007e-03
            5.41474950946129407469697980559431e-01 5.51037760750079045474203454091366e-03 1.24371131542507190430313368523230e-03
            4.53014671446368977214547157927882e-01 5.41474950946130295648117680684663e-01 1.24371131542505607495141539686756e-03
            4.53014671446368533125337307865266e-01 5.51037760750075576027251500477178e-03 1.24371131542505542443011190556490e-03
            5.51037760750074274984644517871857e-03 5.41474950946130850759629993262934e-01 1.24371131542505542443011190556490e-03
            5.51037760750075489291077701636823e-03 4.53014671446370031926420551826595e-01 1.24371131542507038642009220552609e-03
            8.70847043597263614778114515502239e-01 1.02747376222606420781424674260052e-01 1.62350620686912841886262448554135e-03
            8.70847043597263614778114515502239e-01 2.64055801801296730069168461341178e-02 1.62350620686911649263872714499257e-03
            1.02747376222606059958941671084176e-01 8.70847043597264613978836678143125e-01 1.62350620686912559993697602322982e-03
            1.02747376222606198736819749228744e-01 2.64055801801302836295803899702150e-02 1.62350620686912711782001750293603e-03
            2.64055801801301101572327922895056e-02 8.70847043597263170688904665439622e-01 1.62350620686910998742569223196597e-03
            2.64055801801291629982149089528320e-02 1.02747376222606198736819749228744e-01 1.62350620686911302319177519137838e-03
            8.08055925125646168361015497794142e-01 1.63625151528200618589181658535381e-01 2.10591663564597139993161967197466e-03
            8.08055925125645946316410572762834e-01 2.83189233461530534552430538042245e-02 2.10591663564596185895250179953564e-03
            1.63625151528201479012025743031700e-01 8.08055925125645280182595797668910e-01 2.10591663564601043120982915013428e-03
            1.63625151528200368789001117875159e-01 2.83189233461531228441820928765082e-02 2.10591663564597703778291659659772e-03
            2.83189233461529042690241197988144e-02 8.08055925125646723472527810372412e-01 2.10591663564593844018557611263986e-03
            2.83189233461536050973084144288805e-02 1.63625151528201173700693971113651e-01 2.10591663564601043120982915013428e-03
            7.37116338815208438361992193676997e-01 2.34175394401963360424545612659131e-01 2.39183700229278874485605577149272e-03
            7.37116338815208438361992193676997e-01 2.87082667828285863220738605150473e-02 2.39183700229278744381344878888740e-03
            2.34175394401964304114116544042190e-01 7.37116338815206995072060180973494e-01 2.39183700229280175528212559754593e-03
            2.34175394401963415935696843916958e-01 2.87082667828283261135524639939831e-02 2.39183700229280739313342252216898e-03
            2.87082667828280589661371635656906e-02 7.37116338815208993473504506255267e-01 2.39183700229279047957953174829981e-03
            2.87082667828285308109226292572203e-02 2.34175394401963304913394381401304e-01 2.39183700229277313234477198022887e-03
            6.59255694951647930324156732240226e-01 3.12168255169341768429092098813271e-01 2.61067552431568095094860382232582e-03
            6.59255694951648818502576432365458e-01 2.85760498790107037025975955657486e-02 2.61067552431569439505554264258080e-03
            3.12168255169341601895638405039790e-01 6.59255694951647819301854269724572e-01 2.61067552431567357837383092089567e-03
            3.12168255169341601895638405039790e-01 2.85760498790105232913560939778108e-02 2.61067552431567964990599683972050e-03
            2.85760498790105232913560939778108e-02 6.59255694951647930324156732240226e-01 2.61067552431570523707726749762514e-03
            2.85760498790103151245389767609595e-02 3.12168255169340769228369936172385e-01 2.61067552431567097628861695568503e-03
            8.21610770110138166621993605076568e-01 1.13243304903654681825564409791696e-01 2.73461244959868725323826055273457e-03
            8.21610770110138166621993605076568e-01 6.51459249862066797076565194402065e-02 2.73461244959867424281219072668137e-03
            1.13243304903655458981681647401274e-01 8.21610770110138166621993605076568e-01 2.73461244959870460047302032080552e-03
            1.13243304903654931625744950451917e-01 6.51459249862074013526225257919577e-02 2.73461244959867684489740469189201e-03
            6.51459249862070127745639069871686e-02 8.21610770110138166621993605076568e-01 2.73461244959864084938527817314480e-03
            6.51459249862063743963247475221578e-02 1.13243304903654806725654680121806e-01 2.73461244959871240672866221643744e-03
            5.74541105959161191307771332503762e-01 3.96768611430950823848462505338830e-01 2.81085003745577691539048181823546e-03
            5.74541105959161968463888570113340e-01 2.86902826098883213801205016579843e-02 2.81085003745583199286084408186071e-03
            3.96768611430950157714647730244906e-01 5.74541105959161635396981182566378e-01 2.81085003745579122685915862689399e-03
            3.96768611430950157714647730244906e-01 2.86902826098882103578180391423302e-02 2.81085003745578862477394466168334e-03
            2.86902826098882762773101262609998e-02 5.74541105959161080285468869988108e-01 2.81085003745580250256175247614010e-03
            2.86902826098882068883710871887160e-02 3.96768611430949935670042805213598e-01 2.81085003745581377826434632538621e-03
            7.52736321001383545947760467242915e-01 1.78349437979342606563903927963111e-01 3.41109598860112311208281354879546e-03
            7.52736321001383879014667854789877e-01 6.89142410192738613661234126084310e-02 3.41109598860108234608112809382874e-03
            1.78349437979343217186567471799208e-01 7.52736321001382435724735842086375e-01 3.41109598860112918361497946762029e-03
            1.78349437979342689830630774849851e-01 6.89142410192736670770941032060364e-02 3.41109598860112354576368254299723e-03
            6.89142410192731813545208297000499e-02 7.52736321001384656170785092399456e-01 3.41109598860108841761329401265357e-03
            6.89142410192748328112699596204038e-02 1.78349437979342745341782006107678e-01 3.41109598860113395410453840383980e-03
            6.77869789721086690725826429115841e-01 2.52933135839176648218540321977343e-01 3.81789174093829586711934531706447e-03
            6.77869789721087134815036279178457e-01 6.91970744397375631118407568465045e-02 3.81789174093828936190631040403787e-03
            2.52933135839176370662784165688208e-01 6.77869789721086135614314116537571e-01 3.81789174093828329037414448521304e-03
            2.52933135839176481685086628203862e-01 6.91970744397370357559040598971478e-02 3.81789174093829369871500034605560e-03
            6.91970744397366888112088645357289e-02 6.77869789721087134815036279178457e-01 3.81789174093831061226889111992477e-03
            6.91970744397369663669650208248640e-02 2.52933135839176037595876778141246e-01 3.81789174093826681050112270554564e-03
            5.96181882441887966983529167919187e-01 3.34657518177715052054566058359342e-01 4.12799132726444570340440520794800e-03
            5.96181882441889077206553793075727e-01 6.91605993803976332179317410009389e-02 4.12799132726450468400258841938921e-03
            3.34657518177714441431902514523244e-01 5.96181882441888189028134092950495e-01 4.12799132726447953051218675568634e-03
            3.34657518177714385920751283265417e-01 6.91605993803973279065999690828903e-02 4.12799132726445524438352308038702e-03
            6.91605993803974805622658550419146e-02 5.96181882441887966983529167919187e-01 4.12799132726447259161828284845797e-03
            6.91605993803971058619950440515822e-02 3.34657518177714385920751283265417e-01 4.12799132726450034719389847737148e-03
            5.09343076403983130440167315100553e-01 4.21087880063123531915181274598581e-01 4.31484913770886335110388287716887e-03
            5.09343076403983130440167315100553e-01 6.95690435328939205117393385080504e-02 4.31484913770895182200115769433069e-03
            4.21087880063123587426332505856408e-01 5.09343076403982908395562390069244e-01 4.31484913770889457612645045969657e-03
            4.21087880063123032314820193278138e-01 6.95690435328938372450124916213099e-02 4.31484913770891452544642419297816e-03
            6.95690435328939899006783775803342e-02 5.09343076403983130440167315100553e-01 4.31484913770889197404123649448593e-03
            6.95690435328939343895271463225072e-02 4.21087880063123753959786199629889e-01 4.31484913770892493378728005382072e-03
            6.87533646543294296726855918677757e-01 1.87462735345928138697146891900047e-01 4.59534140565974035669283281890785e-03
            6.87533646543294296726855918677757e-01 1.25003618110778008665207039484812e-01 4.59534140565973948933109483050430e-03
            1.87462735345927000718546651114593e-01 6.87533646543294296726855918677757e-01 4.59534140565974816294847471453977e-03
            1.87462735345928527275205510704836e-01 1.25003618110777786620602114453504e-01 4.59534140565977418380061436664619e-03
            1.25003618110777009464484876843926e-01 6.87533646543294296726855918677757e-01 4.59534140565979760256754005354196e-03
            1.25003618110778952354777970867872e-01 1.87462735345928055430420045013307e-01 4.59534140565972387681981103924045e-03
            6.09882363677961891745837874623248e-01 2.64906647850886134420989037607796e-01 5.12640561869847754400408845754100e-03
            6.09882363677961114589720637013670e-01 1.25210988471153278345227022327890e-01 5.12640561869851831000577391250772e-03
            2.64906647850885079709115643709083e-01 6.09882363677961891745837874623248e-01 5.12640561869853045307010575015738e-03
            2.64906647850886411976745193896932e-01 1.25210988471152556700261015976139e-01 5.12640561869848188081277839955874e-03
            1.25210988471152390166807322202658e-01 6.09882363677960448455905861919746e-01 5.12640561869852004472924988931481e-03
            1.25210988471152140366626781542436e-01 2.64906647850885690331779187545180e-01 5.12640561869848708498320632998002e-03
            5.25254911987723116517656762880506e-01 3.49006616007701941928331734743551e-01 5.51303731773003687088863244980530e-03
            5.25254911987722339361539525270928e-01 1.25738472004575940754733665016829e-01 5.51303731773008717786943577721104e-03
            3.49006616007701941928331734743551e-01 5.25254911987722672428446912817890e-01 5.51303731773010539246593353368553e-03
            3.49006616007701941928331734743551e-01 1.25738472004575635443401893098780e-01 5.51303731773005074867644026426206e-03
            1.25738472004575940754733665016829e-01 5.25254911987721340160817362630041e-01 5.51303731773003253407994250778756e-03
            1.25738472004575579932250661840953e-01 3.49006616007701775394878040970070e-01 5.51303731773008631050769778880749e-03
            5.33207332734685901520776951656444e-01 2.72179046638476807018491854250897e-01 6.19587670998193468935300032285340e-03
            5.33207332734684902320054789015558e-01 1.94613620626838929039692516198556e-01 6.19587670998195463867297405613499e-03
            2.72179046638476918040794316766551e-01 5.33207332734684791297752326499904e-01 6.19587670998197111854599583580239e-03
            2.72179046638476918040794316766551e-01 1.94613620626838179639150894217892e-01 6.19587670998192861782083440402857e-03
            1.94613620626838179639150894217892e-01 5.33207332734684902320054789015558e-01 6.19587670998191213794781262436118e-03
            1.94613620626837791061092275413102e-01 2.72179046638477140085399241797859e-01 6.19587670998198759841901761546978e-03
            4.46455240965652422335807614217629e-01 3.57596143740140481881439882272389e-01 6.66828712627658831690879992493137e-03
            4.46455240965651589668539145350223e-01 1.95948615294208261516928359924350e-01 6.66828712627661693984615354224843e-03
            3.57596143740141314548708351139794e-01 4.46455240965651312112782989061088e-01 6.66828712627660306205834572779168e-03
            3.57596143740140592903742344788043e-01 1.95948615294208011716747819264128e-01 6.66828712627661173567572561182715e-03
            1.95948615294208455805957669326745e-01 4.46455240965651645179690376608050e-01 6.66828712627656316341839826122850e-03
            1.95948615294208261516928359924350e-01 3.57596143740141036992952194850659e-01 6.66828712627661086831398762342360e-03
            ];
        
    case  30
        
        % ALG. DEG.:   30
        % PTS CARD.:  190
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.4e-16
        
        xyw=[
            3.33333333333334924653001962724375e-01 3.33333333333335590786816737818299e-01 6.88867417375141344315636615647236e-03
            2.60849100043316489694333881743660e-03 2.60849100043398151801965489937629e-03 5.20230124520507699594981665391913e-05
            2.60849100042975139482348545527657e-03 9.94783017999138041709272783919005e-01 5.20230124519940865146679087605008e-05
            9.94783017999136043307828458637232e-01 2.60849100043205120447176170728198e-03 5.20230124520178915286175436172300e-05
            1.99081215329376323697196937700937e-02 1.99081215329428608262762878666763e-02 4.69287187988729707216473130415579e-04
            1.99081215329434957350684953780728e-02 9.60183756934114285286341328173876e-01 4.69287187988677502881867953377082e-04
            9.60183756934114285286341328173876e-01 1.99081215329420559145834346281845e-02 4.69287187988691651720218889209946e-04
            4.97508711341201725897320784497424e-01 4.97508711341202891631496640911791e-01 1.07490719810773936032655129224622e-03
            4.97508711341204723499487272420083e-01 4.98257731759566523110915881034089e-03 1.07490719810772027836831554736818e-03
            4.98257731759559237272316778444292e-03 4.97508711341201725897320784497424e-01 1.07490719810768515021792701702452e-03
            5.38543920770780878970462879351544e-02 5.38543920770771372685814526448667e-02 1.60172025415143064012990503641731e-03
            5.38543920770712045142936119646038e-02 8.92291215845843921350422078830888e-01 1.60172025415135843226521750182201e-03
            8.92291215845845031573446703987429e-01 5.38543920770740355830064061137818e-02 1.60172025415141546129949023935524e-03
            1.10234628175727419896468006754731e-01 1.10234628175722632059674310767150e-01 3.12988790398811920351329618483760e-03
            1.10234628175712945363784456276335e-01 7.79530743648566559755863636382855e-01 3.12988790398805024825512610675560e-03
            7.79530743648562229886067598272348e-01 1.10234628175719370779539474369813e-01 3.12988790398801295170039260540307e-03
            4.68466155978488285605010332801612e-01 4.68466155978493226097469914748217e-01 3.76714678016052502071420171603222e-03
            4.68466155978494502853948233678238e-01 6.30676880430202230209957292572653e-02 3.76714678016036065566485291356003e-03
            6.30676880430198483207249182669329e-02 4.68466155978490950140269433177309e-01 3.76714678016050507139422798275064e-03
            1.73692667715927590021962600985717e-01 1.73692667715930587624129088908376e-01 4.56713737353077724873129383809101e-03
            1.73692667715933224403812573655159e-01 6.52614664568134550393097015330568e-01 4.56713737353075469732610613959878e-03
            6.52614664568138658218288128409768e-01 1.73692667715934417893564045698440e-01 4.56713737353073648272960838312429e-03
            4.10668319049825147892818222317146e-01 4.10668319049824981359364528543665e-01 5.96463374189550032955509095700108e-03
            4.10668319049825536470876841121935e-01 1.78663361900355255329486681148410e-01 5.96463374189533379610139718352002e-03
            1.78663361900354061839735209105129e-01 4.10668319049820540467266027917503e-01 5.96463374189544568576559768757761e-03
            2.49556756982751537510267780817230e-01 2.49556756982739519346026213497680e-01 6.13293441118918387228164945668141e-03
            2.49556756982747429685076667738031e-01 5.00886486034497702135581675975118e-01 6.13293441118921249521900307399846e-03
            5.00886486034504141429124501883052e-01 2.49556756982744321060607717299717e-01 6.13293441118908412568178079027348e-03
            9.80030809579556794197685576364165e-01 1.60168116844329466719187848866568e-02 1.87972845885872992749768273235134e-04
            9.80030809579556794197685576364165e-01 3.95237873600955656738786814230480e-03 1.87972845885852447118599672926109e-04
            1.60168116844335885196048963052817e-02 9.80030809579558348509920051583322e-01 1.87972845885871420656618169253704e-04
            1.60168116844335885196048963052817e-02 3.95237873600896762876777401629624e-03 1.87972845885797505173508969988916e-04
            3.95237873601129909711948684503113e-03 9.80030809579556794197685576364165e-01 1.87972845885893267330393752168050e-04
            3.95237873600939697282807827605211e-03 1.60168116844335885196048963052817e-02 1.87972845885786636046729802806965e-04
            9.56210240585889215125803275441285e-01 4.00653234502970218011697056681442e-02 2.88717501001577647387619229846223e-04
            9.56210240585890103304222975566518e-01 3.72443596381326911051723449475048e-03 2.88717501001509830541730261543876e-04
            4.00653234503037872227260152158124e-02 9.56210240585880777430816124251578e-01 2.88717501001670238253149491924887e-04
            4.00653234502934343930213856310729e-02 3.72443596381445956450262357861902e-03 2.88717501001578894220117588176322e-04
            3.72443596381283976645693023499462e-03 9.56210240585892323750272225879598e-01 2.88717501001511890525857984002300e-04
            3.72443596381152484606213981521705e-03 4.00653234502949193163168217779457e-02 2.88717501001441254754320553388425e-04
            9.18991377915119977970448417181615e-01 7.64894022503797427203764414116449e-02 4.97929257811117000932787934885937e-04
            9.18991377915119755925843492150307e-01 4.51921983449780993030708842184140e-03 4.97929257810937457053024335351665e-04
            7.64894022503812553992474931874312e-02 9.18991377915118534680516404478112e-01 4.97929257811040130998758712621566e-04
            7.64894022503812553992474931874312e-02 4.51921983449856713710435229813811e-03 4.97929257811057911914387474894284e-04
            4.51921983449772319413328958148668e-03 9.18991377915119755925843492150307e-01 4.97929257810921194020437052785155e-04
            4.51921983450051002739744632208385e-03 7.64894022503812553992474931874312e-02 4.97929257811164814248594545631477e-04
            8.69266446632026346108546022151131e-01 1.25877991283983525772072198378737e-01 6.91379784444627739845024017029118e-04
            8.69266446632019906815003196243197e-01 4.85556208399166768646670888642802e-03 6.91379784444791888053938322400427e-04
            1.25877991283981305326022948065656e-01 8.69266446632028344509990347432904e-01 6.91379784444652351234339437979770e-04
            1.25877991283990742221732261896250e-01 4.85556208399150809190691902017534e-03 6.91379784444793839617848796308408e-04
            4.85556208399082721294259812339078e-03 8.69266446632021017038027821399737e-01 6.91379784444680865751475806746384e-04
            4.85556208399116288193519963556355e-03 1.25877991283984885795277364195499e-01 6.91379784444738653727269284132717e-04
            8.08161929939749112961067112337332e-01 1.86984740735146592882642835320439e-01 8.21744901573339650742433537544684e-04
            8.08161929939749557050276962399948e-01 4.85332932510483885946150905965624e-03 8.21744901573368598940438900513072e-04
            1.86984740735145787970949982081947e-01 8.08161929939749446027974499884294e-01 8.21744901573453600390761764060699e-04
            1.86984740735145787970949982081947e-01 4.85332932510453094604452317639698e-03 8.21744901573251288265375968933313e-04
            4.85332932510489957478316824790454e-03 8.08161929939749557050276962399948e-01 8.21744901573370333663914877320167e-04
            4.85332932510419874649887361783840e-03 1.86984740735145787970949982081947e-01 8.21744901573309618342255689071862e-04
            9.31015414728788681664184423425468e-01 4.79634591905657992061406957873260e-02 8.68173733133502769157563427881996e-04
            9.31015414728788681664184423425468e-01 2.10211260806391006528137666009570e-02 8.68173733133412021435726391160870e-04
            4.79634591905629750763218055453763e-02 9.31015414728788681664184423425468e-01 8.68173733133377435386424103569425e-04
            4.79634591905694421254402470822242e-02 2.10211260806433507253299097783383e-02 8.68173733133652497477583676044333e-04
            2.10211260806381569632428352178977e-02 9.31015414728788681664184423425468e-01 8.68173733133441945415686991083248e-04
            2.10211260806409117041226863875636e-02 4.79634591905722315607896177880320e-02 8.68173733133602841018083839941255e-04
            7.38140363468471782937285752268508e-01 2.56995801448647809284153709086240e-01 9.14823479939534667249034782798844e-04
            7.38140363468471893959588214784162e-01 4.86383508287979889062047078596152e-03 9.14823479939481758183017490182465e-04
            2.56995801448652416709705903485883e-01 7.38140363468467120000582326611038e-01 9.14823479939602646725249623926857e-04
            2.56995801448641814079820733240922e-01 4.86383508287964016342241890811238e-03 9.14823479939379517918152107114338e-04
            4.86383508288007471165315109828953e-03 7.38140363468472560093402989878086e-01 9.14823479939524909429482413258938e-04
            4.86383508288008511999400695913209e-03 2.56995801448649974219051728141494e-01 9.14823479939534016727731291496184e-04
            6.61964863274298598305733776214765e-01 3.33118225448955973533315955137368e-01 9.90369457488317252863430084630636e-04
            6.61964863274293047190610650432063e-01 4.91691127674512024753328276460707e-03 9.90369457488304025596925761476541e-04
            3.33118225448962690382614937334438e-01 6.61964863274292047989888487791177e-01 9.90369457488271065850882202141747e-04
            3.33118225448954308198779017402558e-01 4.91691127674512284961849672981771e-03 9.90369457488354549418163585983166e-04
            4.91691127674503090927426995904170e-03 6.61964863274293269235215575463371e-01 9.90369457488315084459085113621768e-04
            4.91691127674551576448580547662459e-03 3.33118225448962690382614937334438e-01 9.90369457488376016621178798970959e-04
            5.81189175108866140639918285160093e-01 4.13846657841547505007895324524725e-01 1.05033717173681263347406744657064e-03
            5.81189175108862587926239484659163e-01 4.96416704958642721057238134108047e-03 1.05033717173673088463026203953632e-03
            4.13846657841549614431642112322152e-01 5.81189175108864031216171497362666e-01 1.05033717173672437941722712650972e-03
            4.13846657841551113232725356283481e-01 4.96416704958651134466096621622455e-03 1.05033717173686033836965680876574e-03
            4.96416704958626154448042555600296e-03 5.81189175108860256457887771830428e-01 1.05033717173673522143895198155406e-03
            4.96416704958658420304695724212252e-03 4.13846657841552834078413525276119e-01 1.05033717173679441887756969009615e-03
            8.84274577566934505412632461229805e-01 9.11814742788523008298895433654252e-02 1.38104561579689656333758129846956e-03
            8.84274577566933284167305373557610e-01 2.45439481542091414434381135833974e-02 1.38104561579678749259902925672350e-03
            9.11814742788418924890336825228587e-02 8.84274577566945496620576250279555e-01 1.38104561579675735177863415970023e-03
            9.11814742788630977488040230127808e-02 2.45439481542108310641037149935073e-02 1.38104561579706461467431655165683e-03
            2.45439481542071430419937883016246e-02 8.84274577566932062921978285885416e-01 1.38104561579669143228654704103064e-03
            2.45439481542168436156714506068965e-02 9.11814742788630838710162151983241e-02 1.38104561579712402895336875729981e-03
            8.27639190276182601380128289747518e-01 1.46943421195835172143873137429182e-01 1.73450470263756119197740002135788e-03
            8.27639190276168834614622937806416e-01 2.54173885279898627287398227281301e-02 1.73450470263787929689480726835882e-03
            1.46943421195824069913626885863778e-01 8.27639190276191705208930216031149e-01 1.73450470263776025149626835997196e-03
            1.46943421195851520177910742859240e-01 2.54173885279888080168664288294167e-02 1.73450470263785305920223311915151e-03
            2.54173885279870385989209324861804e-02 8.27639190276168834614622937806416e-01 1.73450470263773314644195622236111e-03
            2.54173885279852795893162920037867e-02 1.46943421195844137194796985568246e-01 1.73450470263755165099828214891886e-03
            7.62587568566484019783047187956981e-01 2.12066839505367032980842623146600e-01 1.96352598667988214806157110103868e-03
            7.62587568566486018184491513238754e-01 2.53455919281477641546995727139802e-02 1.96352598667965186352013517989690e-03
            2.12066839505360871243055953527801e-01 7.62587568566489015786658001161413e-01 1.96352598668014886179600253512945e-03
            2.12066839505372944918448752105178e-01 2.53455919281464422954108783869742e-02 1.96352598667935088899705320386602e-03
            2.53455919281491068306699787626712e-02 7.62587568566481244225485625065630e-01 1.96352598667971431356527034495230e-03
            2.53455919281468967929615843104330e-02 2.12066839505368087692716017045313e-01 1.96352598667956252526112237433153e-03
            6.90118710560341575011022996477550e-01 2.84342556304586102644549328033463e-01 2.16174802830700540576258639191565e-03
            6.90118710560342352167140234087128e-01 2.55387331350694947451618332934231e-02 2.16174802830703749814689196284689e-03
            2.84342556304586491222607946838252e-01 6.90118710560341575011022996477550e-01 2.16174802830713117321459471042999e-03
            2.84342556304586491222607946838252e-01 2.55387331350689604503312324368380e-02 2.16174802830678552956200633161643e-03
            2.55387331350701574095296564337332e-02 6.90118710560342352167140234087128e-01 2.16174802830691823590791855735915e-03
            2.55387331350716770272946121167479e-02 2.84342556304585492021885784197366e-01 2.16174802830703966655123693385576e-03
            8.38318567454824137286095719900914e-01 1.01421462494415193233976424380671e-01 2.27676218961402618626288507641675e-03
            8.38318567454823804219188332353951e-01 6.02599700507590388398604375197465e-02 2.27676218961390085249174575210418e-03
            1.01421462494409919674609454887104e-01 8.38318567454828911245101608074037e-01 2.27676218961406521754109455457638e-03
            1.01421462494417274902147596549185e-01 6.02599700507605237631558736666193e-02 2.27676218961397848136729571422165e-03
            6.02599700507544522309899548417889e-02 8.38318567454823804219188332353951e-01 2.27676218961382582570140975519735e-03
            6.02599700507679761352086700298969e-02 1.01421462494414804655917805575882e-01 2.27676218961389564832131782168290e-03
            6.11809416665966332438131303206319e-01 3.62362307698882724071864913639729e-01 2.33811468129887980310632755731604e-03
            6.11809416665957672698539226985304e-01 2.58282756351504681757713655088082e-02 2.33811468129893401321495183253774e-03
            3.62362307698885832696333864078042e-01 6.11809416665963778925174665346276e-01 2.33811468129883860342377310814754e-03
            3.62362307698886387807846176656312e-01 2.58282756351506832814823866328879e-02 2.33811468129894702364102165859094e-03
            2.58282756351497777558279267395847e-02 6.11809416665963112791359890252352e-01 2.33811468129882949612552422991030e-03
            2.58282756351520051407710809598939e-02 3.62362307698885166562519088984118e-01 2.33811468129903722926177245255985e-03
            5.29148852224490484630337050475646e-01 4.44835853963695238011410992839956e-01 2.45289766223974253431250325263591e-03
            5.29148852224484156359096687083365e-01 2.60152938118148047141886536337552e-02 2.45289766223958337343358238058499e-03
            4.44835853963697180901704086863901e-01 5.29148852224488708273497650225181e-01 2.45289766223969179365083093102839e-03
            4.44835853963697180901704086863901e-01 2.60152938118150475754752903867484e-02 2.45289766223965406341522843547409e-03
            2.60152938118142808276989086380127e-02 5.29148852224488042139682875131257e-01 2.45289766223964669084045553404394e-03
            2.60152938118152418645045997891430e-02 4.44835853963697180901704086863901e-01 2.45289766223974990688727615406606e-03
            7.78589579815803345397284829232376e-01 1.59964054253392773397024484438589e-01 2.74691742722695366255014626233333e-03
            7.78589579815792576233945965213934e-01 6.14463659308165585648531248352811e-02 2.74691742722685608435462256693427e-03
            1.59964054253388304749350368183514e-01 7.78589579815803345397284829232376e-01 2.74691742722709981300299730833103e-03
            1.59964054253390774995580159156816e-01 6.14463659308144352633185292233975e-02 2.74691742722679536903296337868596e-03
            6.14463659308128323788267266536423e-02 7.78589579815790133743291789869545e-01 2.74691742722689598299457003349744e-03
            6.14463659308071563636133305408293e-02 1.59964054253384030390705561330833e-01 2.74691742722661409042972380234460e-03
            7.10623785618011427622775499912677e-01 2.27893984493793722023724512837362e-01 3.14303286647104338294433922840199e-03
            7.10623785618027414834330102166859e-01 6.14822298881887718824401645179023e-02 3.14303286647073503584648435094095e-03
            2.27893984493788864797991777777497e-01 7.10623785618015091358756762929261e-01 3.14303286647105856177475402546406e-03
            2.27893984493783979816683427088719e-01 6.14822298881861836750140071217174e-02 3.14303286647066087641788634243767e-03
            6.14822298881921996960286946887209e-02 7.10623785618016312604083850601455e-01 3.14303286647076756191165891607397e-03
            6.14822298881903331335685436442873e-02 2.27893984493780205058399701556482e-01 3.14303286647097746345225210973240e-03
            6.34277654844062399597248713689623e-01 3.03562715120037796001639662790694e-01 3.47181615648817958399985350581574e-03
            6.34277654844071392403748177457601e-01 6.21596300358955855536180479248287e-02 3.47181615648832616813357354601521e-03
            3.03562715120033077553785005875397e-01 6.34277654844068061734674301987980e-01 3.47181615648816136940335574934124e-03
            3.03562715120035075955229331157170e-01 6.21596300358951137088325822332990e-02 3.47181615648845800711774778335439e-03
            6.21596300358957104537083182549395e-02 6.34277654844067839690069376956671e-01 3.47181615648812016972080130017275e-03
            6.21596300359004150237751673557796e-02 3.03562715120029968929316055437084e-01 3.47181615648856729469673432220134e-03
            5.52626687887673773147412248363253e-01 3.84560072995418278374302190059097e-01 3.68914222342292107831296377185026e-03
            5.52626687887675771548856573645025e-01 6.28132391169085035897978741559200e-02 3.68914222342289852690777607335804e-03
            3.84560072995413726459901226917282e-01 5.52626687887679213240232911630301e-01 3.68914222342289722586516909075272e-03
            3.84560072995421997621434684333508e-01 6.28132391169091419680370336209307e-02 3.68914222342312187255530808727144e-03
            6.28132391169068243774731286066526e-02 5.52626687887675771548856573645025e-01 3.68914222342285819458695961259309e-03
            6.28132391169106824024837010256306e-02 3.84560072995415613839043089683400e-01 3.68914222342318172051522928711620e-03
            7.16871672741223320279857489367714e-01 1.71646103407856243094897763512563e-01 3.74822986959171509935284660741672e-03
            7.16871672741206555912185649503954e-01 1.11482223850932246622669197222422e-01 3.74822986959194538389428252855851e-03
            1.71646103407851691180496800370747e-01 7.16871672741222210056832864211174e-01 3.74822986959191806199953589384677e-03
            1.71646103407866429391148699323821e-01 1.11482223850928902075807513938344e-01 3.74822986959156981626173354982257e-03
            1.11482223850929831887590637506946e-01 7.16871672741206555912185649503954e-01 3.74822986959199829296029982117489e-03
            1.11482223850921685626147450420831e-01 1.71646103407862293810381970615708e-01 3.74822986959182525429357113466722e-03
            6.45570482899384456842994950420689e-01 2.42254408443599883460350952191220e-01 4.26090071211563271164068922303159e-03
            6.45570482899385900132926963124191e-01 1.12175108657004585221983461451600e-01 4.26090071211578796939178914726654e-03
            2.42254408443599161815384945839469e-01 6.45570482899384456842994950420689e-01 4.26090071211591633892901143099152e-03
            2.42254408443608210133035640865273e-01 1.12175108657001171286182739095239e-01 4.26090071211536816631060275994969e-03
            1.12175108657009095503021001150046e-01 6.45570482899379571861686599731911e-01 4.26090071211567607972758864320895e-03
            1.12175108657011149415616557689646e-01 2.42254408443608210133035640865273e-01 4.26090071211583047011695057904035e-03
            5.66843738230976668113214600452920e-01 3.19581218448544313037729125426267e-01 4.63114524859060700673341059996346e-03
            5.66843738230963123392314173543127e-01 1.13575043320476631869553330034250e-01 4.63114524859089410346868476153759e-03
            3.19581218448548531885222701021121e-01 5.66843738230973670511048112530261e-01 4.63114524859090277708606464557306e-03
            3.19581218448554693623009370639920e-01 1.13575043320476451458311828446313e-01 4.63114524859076313184624851260196e-03
            1.13575043320475035923955431371724e-01 5.66843738230966565083690511528403e-01 4.63114524859065210954378599694792e-03
            1.13575043320482543807159458992828e-01 3.19581218448554749134160601897747e-01 4.63114524859076226448451052419841e-03
            4.84400341724715965963099506552680e-01 4.01102153748802869870360154891387e-01 4.82765312891821757323995711885800e-03
            4.84400341724701422041476917002001e-01 1.14497504526483842579587246746087e-01 4.82765312891774919790144338094251e-03
            4.01102153748810419386927605955862e-01 4.84400341724708916046893136808649e-01 4.82765312891834420805370342577589e-03
            4.01102153748808587518936974447570e-01 1.14497504526484258913221481179789e-01 4.82765312891774486109275343892477e-03
            1.14497504526481871933718537093227e-01 4.84400341724705918444726648885990e-01 4.82765312891782639309612434885821e-03
            1.14497504526485008313763103160454e-01 4.01102153748809087119298055768013e-01 4.82765312891783506671350423289368e-03
            5.78477356973122636141226848849328e-01 2.45951501392227434017456744186347e-01 5.28119884661372700901393884009849e-03
            5.78477356973136624951337125821738e-01 1.75571141634637689632469914613466e-01 5.28119884661415375098902913464372e-03
            2.45951501392227489528607975444174e-01 5.78477356973121414895899761177134e-01 5.28119884661388747093546669475472e-03
            2.45951501392227489528607975444174e-01 1.75571141634633887118610573452315e-01 5.28119884661426650801496762710485e-03
            1.75571141634641991746690337095060e-01 5.78477356973128076234047512116376e-01 5.28119884661405747383611242184998e-03
            1.75571141634647015505876765928406e-01 2.45951501392224075592807253087813e-01 5.28119884661396206404493369745978e-03
            4.96282829301863182003273777809227e-01 3.25942627323929146054837246992975e-01 5.78190342475091816065724259487979e-03
            4.96282829301863182003273777809227e-01 1.77774543374208421342430597178463e-01 5.78190342475103872393882298297285e-03
            3.25942627323929479121744634539937e-01 4.96282829301863182003273777809227e-01 5.78190342475102831559796712213029e-03
            3.25942627323933198368877128814347e-01 1.77774543374208088275523209631501e-01 5.78190342475116535875256928989074e-03
            1.77774543374205951096200806205161e-01 4.96282829301860906046073296238319e-01 5.78190342475111418441002797408146e-03
            1.77774543374213112034709638464847e-01 3.25942627323925926408065834039007e-01 5.78190342475098581487280569035647e-03
            4.16568600024706914286554138016072e-01 3.31203743439009878901657657479518e-01 6.60339513553086839697803256399311e-03
            4.16568600024701640727187168522505e-01 2.52227656536288813438062561544939e-01 6.60339513553063420930877569503537e-03
            3.31203743439012154858858139050426e-01 4.16568600024703750150933956319932e-01 6.60339513553093605119359565946979e-03
            3.31203743439011766280799520245637e-01 2.52227656536287758726189167646226e-01 6.60339513553067150586350919638789e-03
            2.52227656536286315436257154942723e-01 4.16568600024698143524659599279403e-01 6.60339513553086319280760463357183e-03
            2.52227656536289701616482261670171e-01 3.31203743439003883697324681634200e-01 6.60339513553080334484768343372707e-03
            ];
        
    case  32
        
        % ALG. DEG.:   32
        % PTS CARD.:  210
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.8e-16
        
        xyw=[
            3.19145350604780860329090508287209e-03 3.19145350623165069414710615092190e-03 7.15429619545744280425733641060049e-05
            3.19145350611156012471392173779350e-03 9.93617092987746253207603785995161e-01 7.15429619538814195664477857405927e-05
            9.93617092987416850036197502049617e-01 3.19145350614327000249303978307580e-03 7.15429619598324291083676906843891e-05
            1.64655689217531008006201176385730e-02 1.64655689217531008006201176385730e-02 2.77498317803319188498689662480956e-04
            1.64655689225811190101733672008777e-02 9.67068862156493791459865860815626e-01 2.77498317813518657806021172618216e-04
            9.67068862156493791459865860815626e-01 1.64655689227284109110716059376500e-02 2.77498317816092228502850014493220e-04
            4.98495116352493869427142669792374e-02 4.98495116355436237998155490913632e-02 1.42410433749221628646486337288479e-03
            4.98495116350614331235391318841721e-02 9.00300976729200952419773784640711e-01 1.42410433748367277334567759794481e-03
            9.00300976729124347031074648839422e-01 4.98495116353910652784442447682522e-02 1.42410433749098051282866439493091e-03
            4.88219324360620376968000755368848e-01 4.88219324361344630958114976237994e-01 2.11768290867760458628943354142393e-03
            4.88219324361224116248791915495531e-01 2.35613512779833944588148142429418e-02 2.11768290866827654447823725547551e-03
            2.35613512780159378712241391440330e-02 4.88219324361224116248791915495531e-01 2.11768290867106164301891801926558e-03
            9.41148599404452168570855974394362e-02 9.41148599402267110880515588178241e-02 2.32182957230405708978770995543073e-03
            9.41148599404054153616527855774621e-02 8.11770280118929044022024754667655e-01 2.32182957232449299969645473140645e-03
            8.11770280119172182864417663950007e-01 9.41148599406814168055745994934114e-02 2.32182957230752393465444960440891e-03
            1.56836966604454702833848500631575e-01 1.56836966605318151035675100501976e-01 3.87743907044663885935742442256924e-03
            1.56836966604974537009553614552715e-01 6.86326066790763578140399658877868e-01 3.87743907042447429750486875832394e-03
            6.86326066790314603949241245572921e-01 1.56836966604918914836019894210040e-01 3.87743907044103526884915034145251e-03
            4.47917974916116512495989354647463e-01 4.47917974915863603690979743987555e-01 4.43599992283136117243724783065772e-03
            4.47917974916230809956374514513300e-01 1.04164050167794647316910072731844e-01 4.43599992283645085111576378267273e-03
            1.04164050167938004864964796070126e-01 4.47917974915831462734416845705709e-01 4.43599992283352177052657694389382e-03
            2.27359360510421992795926371400128e-01 2.27359360510134639321577765258553e-01 5.19737966950634556334565061774811e-03
            2.27359360510365593466275413447875e-01 5.45281278978972272497571793792304e-01 5.19737966952780756219043567512017e-03
            5.45281278979131367457000578724546e-01 2.27359360510439229008383676955418e-01 5.19737966951447534491581592419607e-03
            3.84316624257439620748755260137841e-01 3.84316624257142525067365568247624e-01 5.97962751142341181198602839685918e-03
            3.84316624257420691446185401218827e-01 2.31366751484876870259554948461300e-01 5.97962751142215413746594521171573e-03
            2.31366751485209160010825257813849e-01 3.84316624257017791510548931910307e-01 5.97962751142334415777046530138250e-03
            3.06547093375159840356047880050028e-01 3.06547093374228252216084911196958e-01 6.24665603812866489230604116755785e-03
            3.06547093374736845383665695408126e-01 3.86905813249718177893043957737973e-01 6.24665603813135024424685326494000e-03
            3.86905813250220442789384378556861e-01 3.06547093374826551404055408056593e-01 6.24665603812946286510499049882128e-03
            9.78394604898897179978689564450178e-01 1.78715560665132847106839619755192e-02 1.82652260464442357931411087257345e-04
            9.78394604897416142463839605625253e-01 3.73383903520391842098535839511442e-03 1.82652260496885942425704385527752e-04
            1.78715560663958682174889958105268e-02 9.78394604898897179978689564450178e-01 1.82652260468880403999208938969900e-04
            1.78715560663392850071495843167213e-02 3.73383903487901165282880633355944e-03 1.82652260480204678850385535682221e-04
            3.73383903506577675274202832156334e-03 9.78394604898398134729120556585258e-01 1.82652260489532097245332642465598e-04
            3.73383903503422213271401020051599e-03 1.78715560667996875565677328268066e-02 1.82652260485780134312283617958883e-04
            9.57376599458149901700210193666862e-01 3.94715791576244062754952324212354e-02 1.86216417287094317470352855536930e-04
            9.57376599458149901700210193666862e-01 3.15182138377239143278707445006148e-03 1.86216417260169511769390959798898e-04
            3.94715791577081517860214887605252e-02 9.57376599458149901700210193666862e-01 1.86216417286485212689850499145905e-04
            3.94715791578275077000625969958492e-02 3.15182138415708978235185711014310e-03 1.86216417287111285234352253681323e-04
            3.15182138402951127903461880919167e-03 9.57376599458149901700210193666862e-01 1.86216417279412907708618929447653e-04
            3.15182138379868290178897893838439e-03 3.94715791578275077000625969958492e-02 1.86216417266721264182557926680772e-04
            9.29763016779499351649462823843351e-01 6.62735149105910109623707171522256e-02 3.53016453889518197397506815704560e-04
            9.29763016779499351649462823843351e-01 3.96346830996999622415133401887033e-03 3.53016453892536995716466829975388e-04
            6.62735149105613263742498020292260e-02 9.29763016779510564902011537924409e-01 3.53016453884299498660465088661908e-04
            6.62735149105613263742498020292260e-02 3.96346830994368046902076585524810e-03 3.53016453886154297527044665372387e-04
            3.96346830989261888350538853842409e-03 9.29763016779499351649462823843351e-01 3.53016453883206406030165203091542e-04
            3.96346830995000353609070131710723e-03 6.62735149105613263742498020292260e-02 3.53016453889113302096192104073680e-04
            8.85614537945472846125483101786813e-01 1.10207799879526918696015513887687e-01 5.56181907910038592773860344209425e-04
            8.85614537945883739666896872222424e-01 4.17766217511278056817580761617137e-03 5.56181907916344529449470535098499e-04
            1.10207799878525483650015814873768e-01 8.85614537946382562871860955056036e-01 5.56181907910333604184993649965918e-04
            1.10207799879031217993308189306845e-01 4.17766217501465420003370354606886e-03 5.56181907908455007080728016433113e-04
            4.17766217500311135002455387166265e-03 8.85614537945945135000158643379109e-01 5.56181907905549670519118610201303e-04
            4.17766217508616297116041948811471e-03 1.10207799878877507615548836383823e-01 5.56181907912476312938476752378847e-04
            8.27604925831065152408427820773795e-01 1.68070571856776151964396603943896e-01 6.95916250330933482126649636256843e-04
            8.27604925832203464075575993774692e-01 4.32450231215218087316154438326521e-03 6.95916250327329702525525068068646e-04
            1.68070571855081146717125761824718e-01 8.27604925832789106721065763849765e-01 6.95916250324737375131112226966934e-04
            1.68070571856045070102680938362028e-01 4.32450231217714354398085063735380e-03 6.95916250332403985533191725920688e-04
            4.32450231217312765913396432893023e-03 8.27604925831887383580465211707633e-01 6.95916250330972838665510860067798e-04
            4.32450231213263053958728576731119e-03 1.68070571855577416409133206798288e-01 6.95916250325612543124742526146065e-04
            9.40702526955874196445961388235446e-01 4.04066851816024183441378170300595e-02 7.07183221844829082436412193146680e-04
            9.40702526956985640715913632448064e-01 1.88907878617686902922301328544563e-02 7.07183221861312316484926565607338e-04
            4.04066851817778405209224956706748e-02 9.40702526955874196445961388235446e-01 7.07183221842790890772356693361189e-04
            4.04066851806516164069549290616123e-02 1.88907878624055107497081706924291e-02 7.07183221845938112838647615632226e-04
            1.88907878620161624738660322009309e-02 9.40702526957222451287066178338137e-01 7.07183221850435708710769233675819e-04
            1.88907878617485813776966097066179e-02 4.04066851815257505053935460637149e-02 7.07183221847558669825861699109737e-04
            7.61456947234849690353541973308893e-01 2.34130448559401543873192963474139e-01 7.75808288311475289546204070489921e-04
            7.61456947235687353625621653918643e-01 4.41260420569694264414373208182951e-03 7.75808288312376695232408518876355e-04
            2.34130448558233172917653064359911e-01 7.61456947236077374974172471411293e-01 7.75808288313641417066612859798624e-04
            2.34130448558700909877927642810391e-01 4.41260420575276778032414171093478e-03 7.75808288314660133427880239764818e-04
            4.41260420575270793236422051109002e-03 7.61456947235597203516022091207560e-01 7.75808288316887626791251708624486e-04
            4.41260420568741727753714343407410e-03 2.34130448558700909877927642810391e-01 7.75808288312405968691065627496073e-04
            6.91098245011027567663575155165745e-01 3.04467727385770547954280118574388e-01 8.18011073263761375601310632532659e-04
            6.91098245010800527055039310653228e-01 4.43402760321616289107549135906083e-03 8.18011073275978057260660047944612e-04
            3.04467727385771491643851049957448e-01 6.91098245011001255377891538955737e-01 8.18011073279058709313560360243400e-04
            3.04467727385450193100524529654649e-01 4.43402760318876033168722372579396e-03 8.18011073266388289045025761225816e-04
            4.43402760319638097191718983935971e-03 6.91098245011091405487491101666819e-01 8.18011073272017900405439494448956e-04
            4.43402760321932529197219707839395e-03 3.04467727386134423550601013630512e-01 8.18011073276132881330890977977788e-04
            6.16960568882195192763617797027109e-01 3.78576515897221910211811746194144e-01 8.74557017571495998002617522359969e-04
            6.16960568881680049280191724392353e-01 4.46291522062344792054489062138600e-03 8.74557017576378919326840488679409e-04
            3.78576515897503240726251760861487e-01 6.16960568881862014833927787549328e-01 8.74557017578360515637492245133444e-04
            3.78576515897201648641612337087281e-01 4.46291522057206888063340954886371e-03 8.74557017570575185097525583444167e-04
            4.46291522058339315548458614557603e-03 6.16960568881680049280191724392353e-01 8.74557017574167905836490799487137e-04
            4.46291522062590081953992182661750e-03 3.78576515897704413138313839226612e-01 8.74557017574226019072936022524800e-04
            5.38216147161096669648827628407162e-01 4.57285169383518652619358135780203e-01 9.28120841315431435551708272413407e-04
            5.38216147160986979613994662940968e-01 4.49868345537645947934679568902538e-03 9.28120841310121555411960514447856e-04
            4.57285169383451983726729395129951e-01 5.38216147161161395651163275033468e-01 9.28120841312904377128079058678622e-04
            4.57285169383451983726729395129951e-01 4.49868345537558951552359332026754e-03 9.28120841312026607049234794288850e-04
            4.49868345538401940425510261434283e-03 5.38216147160670788096581418358255e-01 9.28120841311737667170267407357187e-04
            4.49868345538044934334154234534253e-03 4.57285169383377654295230740899569e-01 9.28120841307766343032670253165861e-04
            8.99401392370284491484255795512581e-01 7.94579312680868260487798693247896e-02 1.10489500742249474596334835041489e-03
            8.99401392369924779224277244793484e-01 2.11406763620120320690265458551949e-02 1.10489500743311884305153380836373e-03
            7.94579312677945598375472968655231e-02 8.99401392370285601707280420669122e-01 1.10489500742208405018041084133529e-03
            7.94579312671745419116575703810668e-02 2.11406763616783927028919265467266e-02 1.10489500742391244872409039601280e-03
            2.11406763615650632182063617392487e-02 8.99401392370791197272694716957631e-01 1.10489500742459159296493531599026e-03
            2.11406763619169414669674011975076e-02 7.94579312680107063826540070294868e-02 1.10489500742499665089657590044681e-03
            8.45458162389810419234947858058149e-01 1.32357395969260355217400615401857e-01 1.45756221269324032739633789645950e-03
            8.45458162389776668454999253299320e-01 2.21844416412726834486601745766166e-02 1.45756221270069812045999668725926e-03
            1.32357395968531993402095281453512e-01 8.45458162390312684131288278877037e-01 1.45756221269345781835213848864896e-03
            1.32357395968369484506865774164908e-01 2.21844416410163190744864181169760e-02 1.45756221270016447615069932197684e-03
            2.21844416409958632152576996077187e-02 8.45458162390093193039319885428995e-01 1.45756221269965880425745208270882e-03
            2.21844416411478527473288835381027e-02 1.32357395968795726881594987389690e-01 1.45756221269226714752631490767953e-03
            7.82095750515384047218958585290238e-01 1.95018153177622444172101268122788e-01 1.70149578691277530984038435235561e-03
            7.82095750516188403800299511203775e-01 2.28860963067741354781681195618148e-02 1.70149578690138598285885862537725e-03
            1.95018153176744118981744691154745e-01 7.82095750516571985855307502788492e-01 1.70149578690019682991607652411403e-03
            1.95018153176870295828493340195564e-01 2.28860963070662316865000462939861e-02 1.70149578692456991159398516089141e-03
            2.28860963070526349238953400799801e-02 7.82095750515546139780553858145140e-01 1.70149578692208123392726193401359e-03
            2.28860963066924646969191314838099e-02 1.95018153176893555000859237225086e-01 1.70149578689616511571747192732573e-03
            7.13726302839667536836998351645889e-01 2.63119135268269244143368723598542e-01 1.81118911230446948470995671698347e-03
            7.13726302839667536836998351645889e-01 2.31545618919372885036178644213578e-02 1.81118911232448819362272907085298e-03
            2.63119135268292392293432158112410e-01 7.13726302839753135032196951215155e-01 1.81118911233000222903155584930346e-03
            2.63119135267934012301083157581161e-01 2.31545618920304466237247709159419e-02 1.81118911231498429421915563608536e-03
            2.31545618920493134762494946699007e-02 7.13726302839617243733982832054608e-01 1.81118911231023787394844859477416e-03
            2.31545618919266234236875590113414e-02 2.63119135268292392293432158112410e-01 1.81118911232690791603128221964880e-03
            8.53648252343191416002810001373291e-01 9.44179770164805531873497557171504e-02 1.84131161112195836614124910823875e-03
            8.53648252342703695028092170105083e-01 5.19337706409965516041182809203747e-02 1.84131161111245012992898573145339e-03
            9.44179770156613751286300839637988e-02 8.53648252343519042817376885068370e-01 1.84131161111027695509445578636587e-03
            9.44179770163206116828646941030456e-02 5.19337706404461307840847439365461e-02 1.84131161112267112064944107885367e-03
            5.19337706404022908523998580676562e-02 8.53648252342816049598184235946974e-01 1.84131161112546315808402574987213e-03
            5.19337706408202065544443826183851e-02 9.44179770164746412497436267585726e-02 1.84131161111280466403938849140332e-03
            6.42663057493970235611868702108040e-01 3.34082493247154554971700690657599e-01 1.90055366989055592522039894731734e-03
            6.42663057493039424628022970864549e-01 2.32544492590843546175882750048913e-02 1.90055366990668646830220378518561e-03
            3.34082493247783773870906998126884e-01 6.42663057493066514069823824684136e-01 1.90055366990886636519020314040063e-03
            3.34082493247255973845000198707567e-01 2.32544492588073331562625867263705e-02 1.90055366989180774504875071073684e-03
            2.32544492588607973337921919210203e-02 6.42663057493850664592116572748637e-01 1.90055366988493281907302012712080e-03
            2.32544492591022777805420673757908e-02 3.34082493247969403160624324300443e-01 1.90055366991475076406115096716576e-03
            5.67421425923325672968644539650995e-01 4.09119706855485432139829526931862e-01 2.04489977380088163924631849965863e-03
            5.67421425923091859999658481683582e-01 2.34588672212585509779803061292114e-02 2.04489977377964472077254143300706e-03
            4.09119706855425480096499768478679e-01 5.67421425923261169010913818055997e-01 2.04489977378234264945855436224065e-03
            4.09119706855585518745499484793982e-01 2.34588672211401734479796488130887e-02 2.04489977379145298347351555889873e-03
            2.34588672211915282017624662103117e-02 5.67421425923561151272167535353219e-01 2.04489977378876459576662050210416e-03
            2.34588672212738789946140371966976e-02 4.09119706855986309257389166305074e-01 2.04489977378628480855771165636270e-03
            7.95883644812691470882271005393704e-01 1.49641910459035770708524637484516e-01 2.30692764059797338765944729743751e-03
            7.95883644812847901306440689950250e-01 5.44744447286502411764708142527525e-02 2.30692764059193221315435806673122e-03
            1.49641910457888716035057541375863e-01 7.95883644813761947922614581329981e-01 2.30692764058168823734784602663694e-03
            1.49641910458972571262847850448452e-01 5.44744447285333069364021696401323e-02 2.30692764061676651443644203709482e-03
            5.44744447284796137753737355069461e-02 7.95883644812258372880364731827285e-01 2.30692764060293773256682392513994e-03
            5.44744447283740107490501713982667e-02 1.49641910458703203401498171842832e-01 2.30692764057967032026441600578437e-03
            7.31688847721034707483056536148069e-01 2.12466086260515274464211188387708e-01 2.60134378498070228599714504014173e-03
            7.31688847722387958327772139455192e-01 5.58450660178026955793306740361004e-02 2.60134378496625203944225823704528e-03
            2.12466086259354897114448590400571e-01 7.31688847722972823817144671920687e-01 2.60134378497374951430542999730733e-03
            2.12466086260680087072216792876134e-01 5.58450660185428257586970346437738e-02 2.60134378499015349317513567939386e-03
            5.58450660185079370001481890994910e-02 7.31688847720892487913602053595241e-01 2.60134378497148396544580428724203e-03
            5.58450660176888491470492681401083e-02 2.12466086259939790359396738494979e-01 2.60134378496041469494559628117258e-03
            6.63321124311019305608283502806444e-01 2.80443055765804138967922654046561e-01 2.80893694153221910175810549503694e-03
            6.63321124311019305608283502806444e-01 5.62358199233307098907630461326335e-02 2.80893694157729112079180389116573e-03
            2.80443055765542181845262348360848e-01 6.63321124311019305608283502806444e-01 2.80893694159724868070204806258516e-03
            2.80443055766081694724078943181667e-01 5.62358199230450495065269933547825e-02 2.80893694152687615345209692918615e-03
            5.62358199231303007570303975626302e-02 6.63321124311254006755689260899089e-01 2.80893694152039045605628864166192e-03
            5.62358199233420064100386071004323e-02 2.80443055765748405772086471188231e-01 2.80893694157928475174657023671898e-03
            7.52105512554053334817183440463850e-01 1.48431512855705355358182373493037e-01 3.05292773112579443953173985448757e-03
            7.52105512553738697612004671100294e-01 9.94629745906276119260525092613534e-02 3.05292773114595973257823224855656e-03
            1.48431512856109171227814158555702e-01 7.52105512553738697612004671100294e-01 3.05292773114451384056100557984337e-03
            1.48431512855414282636701273077051e-01 9.94629745906932954957468950851762e-02 3.05292773113377460120210216132364e-03
            9.94629745905548645623639458790421e-02 7.52105512553738697612004671100294e-01 3.05292773113050941793944481617018e-03
            9.94629745901721429302000387906446e-02 1.48431512855756869706524980756512e-01 3.05292773114104482728992095985632e-03
            5.89842412473401966899189119430957e-01 3.53507535982312048883358102102648e-01 3.05334850528566964164567387740590e-03
            5.89842412472958987912363681971328e-01 5.66500515446290500709558557446144e-02 3.05334850528014845050250869462616e-03
            3.53507535982529985663092020331533e-01 5.89842412472691091096521631698124e-01 3.05334850529165877444648380389935e-03
            3.53507535982596321488813373434823e-01 5.66500515441443891484496475641208e-02 3.05334850527852258092464943217692e-03
            5.66500515442733276749720516818343e-02 5.89842412473763233471402145369211e-01 3.05334850528333383648527110665327e-03
            5.66500515446690111609484574728413e-02 3.53507535982596321488813373434823e-01 3.05334850528771141117689857935602e-03
            5.11476044318352696116392053227173e-01 4.31429040443182809649869113854947e-01 3.22728101070364047781025007566313e-03
            5.11476044318822542500413419475080e-01 5.70949152385017075217454873836687e-02 3.22728101068319069011369748523066e-03
            4.31429040442822320233773325526272e-01 5.11476044318545208788862055371283e-01 3.22728101069058321420657264866350e-03
            4.31429040443462530340923422045307e-01 5.70949152383450203585013582596730e-02 3.22728101069626443359039669189769e-03
            5.70949152384480629329743806010811e-02 5.11476044318667555366175747622037e-01 3.22728101070672611719314382128232e-03
            5.70949152385571492840377061384061e-02 4.31429040443013944727823627545149e-01 3.22728101069391822008913806030250e-03
            6.85135741196404968178512717713602e-01 2.13046127940230894948214768191974e-01 3.60219558356911304791703187788698e-03
            6.85135741196668979213768579938915e-01 1.01818130862661060387353018086287e-01 3.60219558359638333464025627961291e-03
            2.13046127941300844632621647178894e-01 6.85135741196165715116705996479141e-01 3.60219558360072274541541226255958e-03
            2.13046127939914037296986748515337e-01 1.01818130863432915189648042542103e-01 3.60219558356828818690420490611359e-03
            1.01818130863392544704915110287402e-01 6.85135741196353564852472572965780e-01 3.60219558356819104238955020491630e-03
            1.01818130862524822144443703564320e-01 2.13046127940857032978527740851860e-01 3.60219558359784050236007679757222e-03
            6.11188722458989031061094010510715e-01 2.86145793841027562542933537770296e-01 4.03034387042998723527098547947389e-03
            6.11188722458106847845726861123694e-01 1.02665483700396562016798895911052e-01 4.03034387042768092040967431444187e-03
            2.86145793842079221303009717303212e-01 6.11188722457293609480188933957834e-01 4.03034387043032203690184900324311e-03
            2.86145793840845930056104862160282e-01 1.02665483699732676403648667928792e-01 4.03034387043175491849300584590310e-03
            1.02665483699930296102031945792987e-01 6.11188722458957167660287268518005e-01 4.03034387043444027043381794328525e-03
            1.02665483700433671221396991768415e-01 2.86145793841670770252250122211990e-01 4.03034387042637901044095372071752e-03
            5.31047050461047831504401983693242e-01 3.65311627705897923323874465495464e-01 4.32824057954327417324646987140113e-03
            5.31047050460632052981679862568853e-01 1.03641321833449345790612028395117e-01 4.32824057952659047021626292917063e-03
            3.65311627706307484597658685743227e-01 5.31047050459976799352546095178695e-01 4.32824057952477941890734314256406e-03
            3.65311627705974806268329757585889e-01 1.03641321832808885883281391215860e-01 4.32824057954869344938542141676407e-03
            1.03641321833035579547122040366958e-01 5.31047050460862424259289582550991e-01 4.32824057954890421828775259882605e-03
            1.03641321833532973339941918311524e-01 3.65311627705974806268329757585889e-01 4.32824057952255376868766489906193e-03
            6.16755405363580533339984413032653e-01 2.23611895416044620388262842425320e-01 4.44987478291855760581530176978049e-03
            6.16755405364591280381603155547054e-01 1.59632699220018936525633534984081e-01 4.44987478291111911155031322095965e-03
            2.23611895415503914019694775561220e-01 6.16755405364456166239506273996085e-01 4.44987478291454519041536741497112e-03
            2.23611895415582712098867546046677e-01 1.59632699220264823170012391528871e-01 4.44987478291105319205822610229006e-03
            1.59632699220314533405939982912969e-01 6.16755405364404762913466129248263e-01 4.44987478291500575949823925725468e-03
            1.59632699219956236680317829268461e-01 2.23611895415993855440461857142509e-01 4.44987478291243229722162766393012e-03
            5.40939115423062610865656552050496e-01 2.97770116249244931605488773129764e-01 4.94219666042223018664891753815027e-03
            5.40939115423441418961658655462088e-01 1.61290768328309741530546261856216e-01 4.94219666041111494597659614669283e-03
            2.97770116248247340706711838720366e-01 5.40939115423052063746922613063361e-01 4.94219666041329375866242301640341e-03
            2.97770116248895821975395392655628e-01 1.61290768327288530636920427241421e-01 4.94219666042295009689144791309445e-03
            1.61290768327623290634420527567272e-01 5.40939115423788585701458941912279e-01 4.94219666043293343049569443792279e-03
            1.61290768328408801179918441448535e-01 2.97770116248729010965945462885429e-01 4.94219666040348129532056020707387e-03
            4.59837236520724612631738636991940e-01 3.77459830225164549855065843075863e-01 5.24175491534906081575462266641807e-03
            4.59837236521587033877267458592542e-01 1.62702933254249698658000511386490e-01 5.24175491534333536092216121460297e-03
            3.77459830223968673124090855708346e-01 4.59837236521400793964886588582885e-01 5.24175491533929085313792128886234e-03
            3.77459830225057080266282127922750e-01 1.62702933253738080132677623623749e-01 5.24175491535499877421289127710224e-03
            1.62702933254021270270683885428298e-01 4.59837236521242032072365191197605e-01 5.24175491535656956632038827592623e-03
            1.62702933254432968723790509102400e-01 3.77459830224411874155521218199283e-01 5.24175491533734969756830324172370e-03
            4.66649539183050188206891562003875e-01 3.03210043229955872323699850312551e-01 5.77351026521596717017148492345768e-03
            4.66649539182556305494387061116868e-01 2.30140417587519352515812443016330e-01 5.77351026520697089422506920186606e-03
            3.03210043229811321285893654930987e-01 4.66649539182124373226656643964816e-01 5.77351026521219501397297335643088e-03
            3.03210043230049408613524519751081e-01 2.30140417586447293407658776231983e-01 5.77351026521428535576152540897965e-03
            2.30140417586897821911051664756087e-01 4.66649539182707573381492238695500e-01 5.77351026522439098737082829870815e-03
            2.30140417587695655932122917874949e-01 3.03210043229433678924067407933762e-01 5.77351026520226198734953015900828e-03
            ];
        
        
    otherwise
        xyw=[];
end




pointset_stats=[
    1    1 1.92e-16    0    0
    2    3 1.42e-16    0    0
    4    6 1.53e-16    0    0
    5   10 3.05e-16    0    0
    7   15 3.61e-16    0    0
    8   21 5.62e-16    0    0
    10   28 4.72e-16    0    0
    12   36 4.44e-16    0    0
    14   45 5.00e-16    0    0
    15   55 4.44e-16    0    0
    17   66 5.34e-16    0    0
    19   78 8.53e-16    0    0
    20   91 4.30e-16    0    0
    22  105 5.69e-16    0    0
    24  120 6.18e-16    0    0
    25  136 6.38e-16    0    0
    27  153 5.00e-16    0    0
    28  171 5.27e-16    0    0
    30  190 6.42e-16    0    0
    32  210 6.83e-16    0    0
    ];
