
function [S1,S2,S3,pointset_stats]=set_vioreanu_rokhlin_compact(deg)

%--------------------------------------------------------------------
% CUBATURE RULE ON THE SIMPLEX [0 0; 0 1; 1 0] WITH ADE EQUAL TO deg.
% THE CUBATURE RULE IS DESCRIBED IN COMPACT FORM VIA ITS ROTATIONS.
%--------------------------------------------------------------------
% INPUT:
% deg: algebraic degree of precision (ADE) of the rule.
%--------------------------------------------------------------------
% OUTPUT:
% S1,S2,S3,w:
%    S1 is a matrix N1 x 3,
%    S2 is a matrix N2 x 2,
%    S3 is a matrix N3 x 1,
%    with N1, N2, N3 possibly zero.
%    The pointset is written in terms of rotations of type s1, s2, s3
%    applied resp. to S1, S2, S3, where:
%    s1(w,a,b): [a b c w; a c b w; b a c w; b c a w; c a b w; c b a w];
%            with c=1-a-b.
%    s2(w,a): [a a c w; a c a w; c a a w];
%            with c=1-2*a.
%    s3(w): [1/3 1/3 1/3 w];
%    S1 gives points of which one has to consider all the rotations of
%    type s1.
%    S2 gives points of which one has to consider all the rotations of
%    type s2.
%    S3 gives points of which one has to consider all the rotations of
%    type s3.
%    REMARK: the sum of the weights of the set extracted from S1,S2,S3,
%            for each rule, makes 1.
%    REMARK: in S1, S2, S3, the first component is that of the weight w.
%
% pointset_stats:
%    statistics of the pointset stored in a matrix. Each rows contains
%    degree, cardinality of the full pointset, moment error w.r.t. orthn.
%    Dubiner basis, number of points with negative weights, number of
%    points outside the domain.
%--------------------------------------------------------------------
% SOURCE:
%  B. Vioreanu and V. Rokhlin,
%  "Spectra of Multiplication Operators as a Numerical Tool",
%  Methods and Algorithms for Scientific Computing,
%  SIAM J. Sci. Comput., 36,1 (2014), pp.267-288.
%--------------------------------------------------------------------
% | DEG | CARD | W- | PT.O|   M.E.   |Q.|
%--------------------------------------------------------------------
% |   1 |   1 |   0 |   0 | 1.92e-16 |PI|
% |   2 |   3 |   0 |   0 | 4.20e-16 |PI|
% |   4 |   6 |   0 |   0 | 1.53e-16 |PI|
% |   5 |  10 |   0 |   0 | 4.43e-16 |PI|
% |   7 |  15 |   0 |   0 | 6.11e-16 |PI|
% |   8 |  21 |   0 |   0 | 5.27e-16 |PI|
% |  10 |  28 |   0 |   0 | 7.42e-16 |PI|
% |  12 |  36 |   0 |   0 | 7.49e-16 |PI|
% |  14 |  45 |   0 |   0 | 1.03e-15 |PI|
% |  15 |  55 |   0 |   0 | 1.62e-15 |PI|
% |  17 |  66 |   0 |   0 | 1.55e-15 |PI|
% |  19 |  78 |   0 |   0 | 1.40e-15 |PI|
% |  20 |  91 |   0 |   0 | 5.97e-16 |PI|
% |  22 | 105 |   0 |   0 | 3.03e-14 |PI|
% |  24 | 120 |   0 |   0 | 2.02e-14 |PI|
% |  25 | 136 |   0 |   0 | 2.12e-14 |PI|
% |  27 | 153 |   0 |   0 | 2.52e-13 |PI|
% |  28 | 171 |   0 |   0 | 2.86e-15 |PI|
% |  30 | 190 |   0 |   0 | 2.90e-14 |PI|
% |  32 | 210 |   0 |   0 | 4.08e-12 |PI|
%--------------------------------------------------------------------
% LEGEND:
%--------------------------------------------------------------------
% DEG  : algebraic degree of precision
% CARD : cardinality of the pointset
% W-   : negative weights
% PT.O : points outside the simplex [0 0; 1 0; 0 1].
% M.E. : maximum moment error in inf norm.
% Q.: quality of the rule.
%        PI: pos. weights, points in simplex
%        NI: some neg. weights, points in simplex
%        P0: pos. weights, some points out of the simplex
%        N0: some neg. weights, some points out of the simplex
%--------------------------------------------------------------------
% REMARK:
% There are some issues about the degrees of precision of these rules.
% Here we list for rules of degree ( X) the error at the actual degree
% (first row) and at next one, so to show that the degrees of
% precision are those written in the file.
%--------------------------------------------------------------------
% (  1) deg:   1 cub.err.: 1.92e-16
% (  1) deg:   2 cub.err.: 6.80e-01
%--------------------------------------
% (  2) deg:   2 cub.err.: 4.20e-16
% (  2) deg:   3 cub.err.: 2.05e-01
%--------------------------------------
% (  4) deg:   4 cub.err.: 1.53e-16
% (  4) deg:   5 cub.err.: 3.01e-01
%--------------------------------------
% (  5) deg:   5 cub.err.: 4.43e-16
% (  5) deg:   6 cub.err.: 1.77e-01
%--------------------------------------
% (  7) deg:   7 cub.err.: 6.11e-16
% (  7) deg:   8 cub.err.: 2.42e-01
%--------------------------------------
% (  8) deg:   8 cub.err.: 5.27e-16
% (  8) deg:   9 cub.err.: 1.52e-01
%--------------------------------------
% ( 10) deg:  10 cub.err.: 7.42e-16
% ( 10) deg:  11 cub.err.: 1.91e-01
%--------------------------------------
% ( 12) deg:  12 cub.err.: 7.49e-16
% ( 12) deg:  13 cub.err.: 1.92e-01
%--------------------------------------
% ( 14) deg:  14 cub.err.: 1.03e-15
% ( 14) deg:  15 cub.err.: 1.65e-01
%--------------------------------------
% ( 15) deg:  15 cub.err.: 1.62e-15
% ( 15) deg:  16 cub.err.: 2.66e-01
%--------------------------------------
% ( 17) deg:  17 cub.err.: 1.55e-15
% ( 17) deg:  18 cub.err.: 2.35e-01
%--------------------------------------
% ( 19) deg:  19 cub.err.: 1.40e-15
% ( 19) deg:  20 cub.err.: 3.26e-01
%--------------------------------------
% ( 20) deg:  20 cub.err.: 5.97e-16
% ( 20) deg:  21 cub.err.: 1.73e-01
%--------------------------------------
% ( 22) deg:  22 cub.err.: 3.03e-14
% ( 22) deg:  23 cub.err.: 1.09e-01
%--------------------------------------
% ( 24) deg:  24 cub.err.: 2.02e-14
% ( 24) deg:  25 cub.err.: 1.62e-01
%--------------------------------------
% ( 25) deg:  25 cub.err.: 2.12e-14
% ( 25) deg:  26 cub.err.: 6.77e-02
%--------------------------------------
% ( 27) deg:  27 cub.err.: 2.52e-13
% ( 27) deg:  28 cub.err.: 1.31e-01
%--------------------------------------
% ( 28) deg:  28 cub.err.: 2.86e-15
% ( 28) deg:  29 cub.err.: 5.42e-02
%--------------------------------------
% ( 30) deg:  30 cub.err.: 2.90e-14
% ( 30) deg:  31 cub.err.: 7.90e-02
%--------------------------------------
% ( 32) deg:  32 cub.err.: 4.08e-12
% ( 32) deg:  33 cub.err.: 1.04e-01
%--------------------------------------------------------------------


switch deg
    
    case   1
        
        % ALG. DEG.:    1
        % PTS CARD.:    1
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.9e-16
        
        S1=[];
        
        S2=[];
        
        S3=[
            2.00000000000000000000000000000000e+00
            ];
        
        
    case   2
        
        % ALG. DEG.:    2
        % PTS CARD.:    3
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.2e-16
        
        S1=[];
        
        S2=[
            6.66666666666666740681534975010436e-01 1.66666666666666601903656896865868e-01
            ];
        
        S3=[];
        
        
    case   4
        
        % ALG. DEG.:    4
        % PTS CARD.:    6
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.5e-16
        
        S1=[];
        
        S2=[
            2.19903487310643713792401854334457e-01 9.15762135097707430375635340169538e-02
            4.46763179356022888111255042531411e-01 4.45948490915964890213274429697776e-01
            ];
        
        S3=[];
        
        
    case   5
        
        % ALG. DEG.:    5
        % PTS CARD.:   10
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.4e-16
        
        S1=[
            2.06732562772980604792749659281981e-01 6.14926166643125493571631068334682e-01 3.13979397676981020204323158395709e-01
            ];
        
        S2=[
            1.20557960814281106554091138605145e-01 7.11129945334409985235879503306933e-02
            ];
        
        S3=[
            3.97930740919273107092379859750508e-01
            ];
        
        
    case   7
        
        % ALG. DEG.:    7
        % PTS CARD.:   15
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.1e-16
        
        S1=[
            1.09431235362070694572267370858754e-01 7.18483300897606480539536732976558e-01 2.33346892984868908849094282231817e-01
            ];
        
        S2=[
            6.40266694597593916116196055554610e-02 5.12530798414517832495285176719335e-02
            1.29759675094103410497581307936343e-01 4.74110794051135320259504624118563e-01
            2.54017851388662285749830971326446e-01 2.42184756344486107337843350251205e-01
            ];
        
        S3=[];
        
        
    case   8
        
        % ALG. DEG.:    8
        % PTS CARD.:   21
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.3e-16
        
        S1=[
            7.68116900762054882756402207633073e-02 7.87642410206076615253323325305246e-01 1.71130424662843894090258345386246e-01
            8.33949947677161268488887913008512e-02 5.84632756543619902522834763658466e-01 3.76058771577794381801851386626367e-01
            ];
        
        S2=[
            2.74290145101195696752949260144305e-02 3.14057488009514829130353064101655e-02
            1.48795218039663695863694670151745e-01 1.96019357807156596740227882946783e-01
            1.70029064429040394079706288721354e-01 4.04461999739164801415114425253705e-01
            ];
        
        S3=[];
        
        
    case  10
        
        % ALG. DEG.:   10
        % PTS CARD.:   28
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 7.4e-16
        
        S1=[
            4.70042084257699527283236307084735e-02 8.38551707380394306667881210159976e-01 1.30053450518161195503807903151028e-01
            5.70522454912146884109525046824274e-02 6.66920601059607931482275944290450e-01 3.02790097367536981121105554848327e-01
            1.18368292937497898709153787422110e-01 5.20560306058802990669676091783913e-01 3.30714245940923079469797585261404e-01
            ];
        
        S2=[
            1.47632657923944792671688830409948e-02 2.27066403128516301757056083943098e-02
            5.29577720029942577983916862649494e-02 4.85348136374768979184324280140572e-01
            1.00218628461750294822429907526384e-01 1.55015517336223013211693455559725e-01
            ];
        
        S3=[
            1.61632520101687493507824910921045e-01
            ];
        
        
    case  12
        
        % ALG. DEG.:   12
        % PTS CARD.:   36
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 7.5e-16
        
        S1=[
            2.96287076580063896136252310498094e-02 8.69373866882279577872338904853677e-01 1.06407946399805095061275039824977e-01
            3.66186449359994781183047507511219e-02 5.62233442343925027273598971078172e-01 4.14262053470467184190795251197414e-01
            3.80516989422833296075054931861814e-02 7.25103351867971257505018911615480e-01 2.52014086602536824255338387956726e-01
            8.30538042395671527451384008600144e-02 6.17412891519986017918597553943982e-01 2.66109864468771106782440938332002e-01
            ];
        
        S2=[
            1.02827004393695707606948275270042e-02 1.93652109024277097049093754321802e-02
            6.32792852569625113856588427552197e-02 1.20430831569190502361443861900625e-01
            9.34833183420127056262316500578891e-02 4.40005690210435618237028165822267e-01
            1.24915651076608999470174410362233e-01 2.71524632000497700978769444191130e-01
            ];
        
        S3=[];
        
        
    case  14
        
        % ALG. DEG.:   14
        % PTS CARD.:   45
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.0e-15
        
        S1=[
            1.64081638009672299560737229739971e-02 8.87822775364344285442541604425060e-01 9.67912764708029893512275521061383e-02
            2.37871838679023506202270255016629e-02 7.67286163241992502648258778208401e-01 2.14478157320871803959150270202372e-01
            2.62517015664536709773635436704353e-02 6.33349939282431551035301708907355e-01 3.48145146811778216644484018615913e-01
            5.78624609428792929799456601358543e-02 7.07553018466495098337531999277417e-01 2.00642015647370303144114700444334e-01
            7.50814215647439553436015557963401e-02 5.41383214828122349970840332389344e-01 3.61345725637031711663382793631172e-01
            ];
        
        S2=[
            9.24582234230646345107818007136302e-03 1.88250991695915192991606090799905e-02
            2.86857736729406211095838585833917e-02 4.90249109938372285011354279049556e-01
            3.47964458789228534008053372872382e-02 8.19725416516456917292998696211725e-02
            9.07721755342435904223563625237148e-02 2.20530835565724586677660568057036e-01
            1.04384585752360203869670840504114e-01 3.86124189278291285454258741083322e-01
            ];
        
        S3=[];
        
        
    case  15
        
        % ALG. DEG.:   15
        % PTS CARD.:   55
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.6e-15
        
        S1=[
            1.13478578707406092124232444007248e-02 9.03429699782029027588237113377545e-01 8.35136443388274640753365929413121e-02
            1.79581564630208086186247129489857e-02 8.10386575943458420390186347503914e-01 1.70743649147708009827795194723876e-01
            2.14331937107664106945126292202985e-02 7.02894663916247330881503785349196e-01 2.80682296607787373776687900317484e-01
            2.34471608606178598399605306212834e-02 5.64636464683940575426390751090366e-01 4.19465101758231984074853926358628e-01
            5.04487066038005402268673549315281e-02 5.84478584036716264371591478266055e-01 3.33042061006166123515725985271274e-01
            5.16399502525189679769113126894808e-02 7.17030852299470722321927951270482e-01 1.92513233859061300146109374509251e-01
            6.91055130456363075897030512351193e-02 4.72245408246903919380343950251699e-01 3.36118950295532514793706013733754e-01
            ];
        
        S2=[
            7.82420081206926590344785665820382e-03 1.76519929574460200660812603246086e-02
            3.30674512743280579751825598577852e-02 7.71703842193709466101836369489320e-02
            4.12329407112786774924906296746485e-02 4.59371797617381893985566421179101e-01
            6.39307974597222994361800374463201e-02 2.04922897588778796240660540206591e-01
            ];
        
        S3=[
            8.95505963851949693399490115552908e-02
            ];
        
        
    case  17
        
        % ALG. DEG.:   17
        % PTS CARD.:   66
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.6e-15
        
        S1=[
            8.91434937396722026914375192063744e-03 9.22032114610694453205042009358294e-01 6.50938297812631017080775563954376e-02
            1.26224534910094991768669814291570e-02 8.42560091151481893589902938401792e-01 1.43419095062200907531035909414641e-01
            1.52913754878598005804146708896951e-02 7.43449702189565297949513933417620e-01 2.43155281870591710724482936711865e-01
            1.68222257049058902811378146679999e-02 6.22112357616232181101167952874675e-01 3.65088603079598394973714903244399e-01
            3.36685239000735631043070839041320e-02 5.17329851757723613303596721380018e-01 4.12641979622127286209121166393743e-01
            3.55210598713636485634914663478412e-02 7.62594448273204661425950234843185e-01 1.66359403543997613761362686091161e-01
            3.87830667036489495913542668859009e-02 6.40898402664594746269699498952832e-01 2.92115957257160607785095862709568e-01
            5.77619143371889132154706203436945e-02 5.57643332782426282001608797145309e-01 2.81112663730934075889678069870570e-01
            ];
        
        S2=[
            4.53315013833891639061235423469043e-03 1.33804426737547093456681679413123e-02
            1.84092209662610993559095362570588e-02 4.92937508753781006287653099207091e-01
            2.41091365796606005300617425746168e-02 6.86030848984773289345184821286239e-02
            4.46885783225142926844419832832500e-02 1.63281281675213801873880470338918e-01
            6.25313391043089000786991960012529e-02 4.17510360602973917831803873923491e-01
            7.36253038155479516690959940206085e-02 2.86487437482899287566340262856102e-01
            ];
        
        S3=[];
        
        
    case  19
        
        % ALG. DEG.:   19
        % PTS CARD.:   78
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.4e-15
        
        S1=[
            5.63377005876726510980034845488262e-03 9.33764288367549766256558996246895e-01 5.69009106573930001360395181109197e-02
            9.55747387425846285646802158453283e-03 8.65595106103016709298003661388066e-01 1.21539313278482399849345085840469e-01
            1.20028497260327004364510727896231e-02 7.80245213926672565030173700506566e-01 2.08294353280180588594561186255305e-01
            1.29426337678534807140717788342954e-02 5.47138472365763073312905362399761e-01 4.41297885394377475964944324005046e-01
            1.35341987861764094513983280876346e-02 6.65889139137802787971054385707248e-01 3.23322608813239320557642031417345e-01
            2.85109549636328798993822886131966e-02 7.97030463292363489102854146040045e-01 1.39071275726619897339375597766775e-01
            3.20075860938072720429659057117533e-02 6.88298973534612223978967904258752e-01 2.54674408859317413789113970778999e-01
            3.27241833742634227744083830202726e-02 5.57143388414403584185663476091577e-01 3.85518010456119386564211026779958e-01
            4.09772697387354792319946739098668e-02 6.12834567107272132346906801103614e-01 2.52827382426030589090970579491113e-01
            4.75963481553459410999806777908816e-02 4.92042973154920293410441445303150e-01 3.66154971066125323186923878893140e-01
            ];
        
        S2=[
            3.57098663712716304571848091597985e-03 1.19057700723167493583876108687036e-02
            1.44321904567915604289396469539497e-02 4.66749672319444475210303835410741e-01
            1.66980490629892001919198918358234e-02 5.44172191454610487681975428131409e-02
            3.77258490801887433341299527000956e-02 1.49494614598017605544555408414453e-01
            5.94655559851059584142163316755614e-02 2.45242132245316712957361460212269e-01
            6.37994983667174858643633683641383e-02 3.74619062916066780655199863758753e-01
            ];
        
        S3=[];
        
        
    case  20
        
        % ALG. DEG.:   20
        % PTS CARD.:   91
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.0e-16
        
        S1=[
            5.46770911902901522727571403947877e-03 9.43408146097875488500505980482558e-01 4.66217699188314904645835667906795e-02
            7.74247385738333418481182945924957e-03 7.95199869823192462270355918008136e-01 1.94613925044525598595512860811141e-01
            8.31246760108328257288246021516898e-03 7.24428132024519610965285210113507e-01 2.65496289314309075191289366557612e-01
            8.62749841998047828361961819609860e-03 8.75117681769159272420210982090794e-01 1.14200622336593296535234287603089e-01
            1.15487571158751807509634801363063e-02 6.21242229278482493981528023141436e-01 3.69462178934587326395444506488275e-01
            1.99995967748047114020959469371519e-02 8.23284162466894287035756860859692e-01 1.21432174389561095551037794848526e-01
            2.30772619513648497013846849768015e-02 7.35502973872642962227530460950220e-01 2.11973966375945788165324756846530e-01
            2.36968620977554313145052589106854e-02 5.24604158534802356150805735524045e-01 4.22476993945888390502574338825070e-01
            2.42155779676480401030325140254718e-02 6.30873630515175976007924418809125e-01 3.19958418028188418258395131488214e-01
            3.46006071509829329913010553809727e-02 6.45556379414073733258305765048135e-01 2.34468421561538797881496520858491e-01
            3.53350270661650020520561099601764e-02 5.39652455990818680220399983227253e-01 3.37729237107069923951030432363041e-01
            5.11820687178499433844791610681568e-02 4.51386500463364981339253745318274e-01 3.28819489995002378623922822953318e-01
            ];
        
        S2=[
            1.98111631167665518463150142736140e-03 8.57077238529913021380934878834523e-03
            1.28169873010148503700289523976608e-02 5.15305405051262876980899818590842e-02
            1.39280240150336902249517123664191e-02 4.94659340413900083799347839885741e-01
            3.33468122379746920169729662575264e-02 4.37325812915608291842062271825853e-01
            3.35028775832722988670120400911401e-02 1.32293995381116497034312828873226e-01
            4.48513958907962931310642318294413e-02 2.15238824344608503658093923149863e-01
            ];
        
        S3=[
            5.58829129411608366284625049047463e-02
            ];
        
        
    case  22
        
        % ALG. DEG.:   22
        % PTS CARD.:  105
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.0e-14
        
        S1=[
            2.74487484476933392418818158375871e-03 9.60353805601490417842569513595663e-01 3.24378361207898827101026029140485e-02
            4.39941759149793231581382357830989e-03 9.12769702185126097226941510598408e-01 7.96405637940901245386271511961240e-02
            6.27582954028471640128827857552096e-03 8.46930929612685123863968783552991e-01 1.44767660510350892888453699924867e-01
            7.96061524609002930941148434840215e-03 7.63800852902583815584591775404988e-01 2.27743920492909590613805903558386e-01
            9.33000760614092344846781656997337e-03 6.63853996353002795416387016302906e-01 3.27598731749432103832475604576757e-01
            1.01534342231912096815582557951529e-02 5.52855567624573107643470848415745e-01 4.38428235397557819474911866564071e-01
            1.17725766836053805097783708788484e-02 8.69853922593129991192029137891950e-01 8.94784746607580894206535049306694e-02
            1.64883938611260598416929923359930e-02 7.92000836049181700104782066773623e-01 1.64479570130591612464598938458948e-01
            1.98517142131345615541082594290856e-02 6.98032087133780287935280739475274e-01 2.57896711320237226328089263915899e-01
            2.22299864957886013361321175807461e-02 5.90921636898512403490713040810078e-01 3.64207294093872813167678259560489e-01
            2.62151931207954894120781830224587e-02 7.13756998577424806029512183158658e-01 1.81577875343950506659140842202760e-01
            3.10609081946875316415646040013598e-02 6.14456089155397067180786052631447e-01 2.79145603763538918684616874088533e-01
            3.42703883173083798108216058153630e-02 5.03163853150140005787704922113335e-01 3.88410709745712179330467961335671e-01
            4.16310557661731875378485767669190e-02 5.18446604434296465235831874451833e-01 2.89296348423922711212696867733030e-01
            ];
        
        S2=[
            9.37274068116606090365905412653547e-04 5.81913467083006175073478516424075e-03
            6.67814979626495976161582035501851e-03 3.70242529014878085646955696574878e-02
            2.01890425898365484524976665170470e-02 1.00139964243320195302899833222909e-01
            2.31597682585548414468945566113689e-02 4.77283361444093778835906505264575e-01
            3.45410556055520193141639140321786e-02 1.87840033887680113444673679623520e-01
            4.42913422734278680592545640593016e-02 4.02759919721537684722534322645515e-01
            4.81012426657117103667715696246887e-02 2.95819836686848580686870491263107e-01
            ];
        
        S3=[];
        
        
    case  24
        
        % ALG. DEG.:   24
        % PTS CARD.:  120
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 2.0e-14
        
        S1=[
            2.52108010911473393728110536926579e-03 9.23455229698260882109650538041024e-01 7.19187624498160982255967610399239e-02
            2.61538266742005788556557632773547e-03 9.63258241935612047335268925962737e-01 2.93101710973713291963349547586404e-02
            4.64376377579891076036178176877911e-03 8.65062183147062846444441674975678e-01 1.27671748918951111217978677814244e-01
            6.23226080856589727957617341758123e-03 7.93149355629101493470045625144849e-01 1.99376854580637002944953906080627e-01
            7.43565906489512557647758583811992e-03 7.03167527087424915066549147013575e-01 2.89355487561616608882530954360845e-01
            7.59356404581463823727505157989981e-03 8.94987050429267849693815151113085e-01 7.57231791336101522915669193025678e-02
            8.11937309530654308109554051497980e-03 6.01693105035377984179945087817032e-01 3.90708026101555416786936802964192e-01
            1.28408535161764307541698570958033e-02 8.22015697438525094753458688501269e-01 1.39604852336156298742153580860759e-01
            1.59667719906910303417912899703879e-02 7.37522761587901753976836971560260e-01 2.23633835367293809515132352316868e-01
            1.79836405986236597120964120222197e-02 6.39399415339909515942906637064880e-01 3.21417139704213317141778816221631e-01
            1.90409828345894406587124336738270e-02 5.33728924897748924216500654438278e-01 4.26438949417778212502838641739800e-01
            2.12182136341521193623194108113239e-02 7.52006738431799326427551477536326e-01 1.55247740375081599850659586081747e-01
            2.50197538145131188891401308183049e-02 6.62157408951895787296848538971972e-01 2.44250018940709912040887274997658e-01
            2.80004103985434307788615626577666e-02 5.60254721210861617031184778170427e-01 3.44727467648918317610906569825602e-01
            3.29152067326296191018819570217602e-02 5.73642212535123152683524949679850e-01 2.57088626651901686059886742441449e-01
            3.68135017511886583618796464634215e-02 4.68278059755923625750995142880129e-01 3.59525370113814912720329175499501e-01
            ];
        
        S2=[
            6.75828635344960732336816455756434e-04 4.68044773847087203222372409072705e-03
            4.85001768506211280995721324416081e-03 3.58515749284044130118509485782852e-02
            8.32445914417666819884455975397941e-03 4.96163793331063474045805605783244e-01
            1.56575127058480000064033532680696e-02 8.10973021217386014214412170986179e-02
            2.72363303717661105196601312172788e-02 1.67282792916418010520374082261696e-01
            2.91932767425463687849251925854333e-02 4.51971022222129481527019834175007e-01
            4.01236544967909711245646064980974e-02 2.62525142845060011076441242039436e-01
            4.26847492090775826323323371980223e-02 3.67318860036334404739477577095386e-01
            ];
        
        S3=[];
        
        
    case  25
        
        % ALG. DEG.:   25
        % PTS CARD.:  136
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 2.1e-14
        
        S1=[
            1.31197209643625807562461460520353e-03 9.68738481181608435122143418993801e-01 2.64387673203308991298765562305562e-02
            2.36350636213819879569153492582245e-03 9.35914465591766275842644517979352e-01 5.82420655518003371753366081975400e-02
            3.86335520248911202384478613680585e-03 8.85737420571854561224256485729711e-01 1.07957673880294896529008497054747e-01
            5.20514149707230339148011921679426e-03 8.16939712186552924322313629090786e-01 1.76404098155251293222889330536418e-01
            6.07972796082266800365978554054891e-03 7.34078265782887817714197353780037e-01 2.59206444374597722024589074862888e-01
            6.57386162210078436873850193933322e-03 6.41753644337056394775231638050172e-01 3.51505881363440519926655269955518e-01
            6.80348727378628660017456653008594e-03 5.45201712627923384957284724805504e-01 4.47959453755785386164234296302311e-01
            7.19121241441081186784867185224357e-03 8.99672974696373417557992979709525e-01 6.92285554476964604120325930125546e-02
            1.04059398937450496569478985975366e-02 8.36754745906231556240584268380189e-01 1.29450436723384493253874438778439e-01
            1.27919279216750694100346308346161e-02 7.60018820808685102718982307123952e-01 2.05092818400489096397620869538514e-01
            1.42905425956894403111130387173944e-02 6.72878720973754451684101240971358e-01 2.92117203365807209802795796349528e-01
            1.51659801829572508241605888201775e-02 5.78766335729513636820797728432808e-01 3.85847444474734901920243146378198e-01
            1.68853900992732910979476912416430e-02 7.76386479467535695064839273982216e-01 1.41173270259469990639544789701176e-01
            2.00886415736123216668751467750553e-02 6.95281183841349070817727806570474e-01 2.20694701488698713509606363913917e-01
            2.24137981002940911423593206563964e-02 6.04556578856260240506514946901007e-01 3.10779394840667910759890446570353e-01
            2.37935975814408291750812196596598e-02 5.06851226677805466458437422261341e-01 4.07374461989678027240557867116877e-01
            2.64705991381611500934400993401141e-02 6.17605377498653074219703285052674e-01 2.30766928655943892589874621990020e-01
            2.99529560621967401445875367471672e-02 5.23270463532956031649234773794888e-01 3.23002638196491276278976556568523e-01
            3.60477304788943131441669720516074e-02 4.31308082067835607542605202979757e-01 3.30276680517265386338721100401017e-01
            ];
        
        S2=[
            8.06753861467465185550174133766177e-04 5.71433885823602327747039453242905e-03
            4.09789211500325659814913947798232e-03 2.72494286004120213340318201744594e-02
            1.21111202659417906829508737587275e-02 7.66556657710376621039927158562932e-02
            1.54524875884162107148034337456011e-02 4.82135684364817795888313867180841e-01
            2.23158555811250811085066914074559e-02 1.49523257795368291755977452339721e-01
            3.12810930388166402726923820409866e-02 4.22530364706222016923931050769170e-01
            3.24470552617810575890544555477391e-02 2.35017663942217097661213642822986e-01
            ];
        
        S3=[
            3.82670185191351375220136787902447e-02
            ];
        
        
    case  27
        
        % ALG. DEG.:   27
        % PTS CARD.:  153
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 2.5e-13
        
        S1=[
            6.03425466908624492240464931569477e-04 9.72577577219754130233297928498359e-01 2.56411289038599408962593173555433e-02
            2.06808555118235702294238187448627e-03 9.44757428656674114186841961782193e-01 4.89478566028448414115814557590056e-02
            3.18639900444684298283504197968341e-03 9.00140112535706693464021554973442e-01 9.43702253266816004639849779778160e-02
            4.21605007547614166196758844762371e-03 8.35926640580004987057805010408629e-01 1.58178278042216791110519125140854e-01
            4.84167596017410530040425697961837e-03 7.61488498658509094241253478685394e-01 2.32483150304507807382137229978980e-01
            5.27148148674672425983978030217258e-03 6.79274857925024622140597330144374e-01 3.14735969095626788583075494898367e-01
            5.69417376683317537355843995783289e-03 5.90173779527698449953732051653787e-01 4.03765902224754891225444453084492e-01
            6.09061433332609009072555750208267e-03 9.08147639531831196713085319061065e-01 6.24216082469765121132354579458479e-02
            8.22287150505142221124099677354025e-03 8.53441207153490366543735490267863e-01 1.17038555323194598534186638971732e-01
            1.00572100681038199138095379225888e-02 7.84561667827341380210270926909288e-01 1.84167433829062893169137282711745e-01
            1.12575751317900898257340358554757e-02 7.08519881821631503093783521762816e-01 2.60168754876502983375274880017969e-01
            1.25417067955737593365483917295933e-02 6.23662680919337741869412639061920e-01 3.44939109327520476711015362525359e-01
            1.30763345643324004757834799761440e-02 8.02460515514371586220931931165978e-01 1.25010019659924004331585933869064e-01
            1.34095059093148494949732096870321e-02 5.31386343848350817253844979859423e-01 4.36826791108323175816252614822588e-01
            1.62079675037046684604735702350808e-02 7.29210554614469153023037506500259e-01 1.95225741421016812937594409049780e-01
            1.84320573910986412746737528323138e-02 6.47395307940602737772906039026566e-01 2.77031342028357274731575898840674e-01
            2.00442217795869792595908620569389e-02 5.56614640862459930659156270849053e-01 3.67086174365103212746674898880883e-01
            2.17851769653381911739931098281886e-02 6.55804993103211053551149234408513e-01 2.07677817553239690839816944389895e-01
            2.45375469620846900498101206267165e-02 5.70923789561736949593750978237949e-01 2.91806857017890386440939209933276e-01
            2.63659430662542895729139047489298e-02 4.78028270406271715842194680590183e-01 3.83097780854546821061035188904498e-01
            3.03033996191359698835210423339959e-02 4.88026817278848390468937168407138e-01 2.97907118579965990434743616788182e-01
            ];
        
        S2=[
            8.86471228587608047269630340281310e-04 6.02872510688590157457156237796880e-03
            3.41677604513524694335169229475468e-03 2.25804480640937610402829704980832e-02
            5.87318828321965573369123347902132e-03 4.96948162575856389189254969096510e-01
            9.05683075728702310924767004962632e-03 6.97453864825942071448139358835761e-02
            1.85207876543841201688600506258808e-02 1.33222990598772211834344147973752e-01
            2.06500354284146515893016982090558e-02 4.61535910661834414270288107218221e-01
            2.63073088530244095228916023643251e-02 2.11399010110215612723294498209725e-01
            3.17492163283966469244923302994721e-02 3.92179397457324996345562340138713e-01
            3.37792062752273677950221042465273e-02 3.02074000493973815473225386085687e-01
            ];
        
        S3=[];
        
        
    case  28
        
        % ALG. DEG.:   28
        % PTS CARD.:  171
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 2.9e-15
        
        S1=[
            9.79150045634533290617174117187460e-04 9.76405945543714515189037683740025e-01 1.90656116400246301856302721944303e-02
            1.54092859731612494972319282737772e-03 9.49626039031084334673948887939332e-01 4.57527920669597021330332609068137e-02
            2.38410382462610594481167147762335e-03 9.09575444421073298073565638333093e-01 8.55585301859539404389920491666999e-02
            3.38460472803553879747595622973222e-03 8.53928600117039415806630131555721e-01 1.40745793360055188037094353603607e-01
            4.07519769287254061268521709848756e-03 7.86232363633783681855504710256355e-01 2.08266834439757991814801130203705e-01
            4.45452058323130006822765381002682e-03 7.10218209945296274909765088523272e-01 2.84299186848130214233520973721170e-01
            4.45623474401297004626743003541378e-03 9.19615743255533413247349017183296e-01 5.56921540915347179634409258142114e-02
            4.75197111856252765033081786327784e-03 6.28121500758709094824894236808177e-01 3.66399643439952127366154854826164e-01
            4.97484526170024338176389733234828e-03 5.41474950946130295648117680684663e-01 4.53014671446368977214547157927882e-01
            6.49402482747645903166100467274191e-03 8.70847043597264058867324365564855e-01 1.02747376222606198736819749228744e-01
            8.42366654258392029419599822404052e-03 8.08055925125646168361015497794142e-01 1.63625151528200701855908505422121e-01
            9.56734800917115150997727113235669e-03 7.37116338815208438361992193676997e-01 2.34175394401963304913394381401304e-01
            1.04427020972627099260066074748465e-02 6.59255694951647930324156732240226e-01 3.12168255169341601895638405039790e-01
            1.09384497983947906463164656543086e-02 8.21610770110138166621993605076568e-01 1.13243304903654806725654680121806e-01
            1.12434001498231891935652981828753e-02 5.74541105959161524374678720050724e-01 3.96768611430950213225798961502733e-01
            1.36443839544044195899452631692839e-02 7.52736321001383545947760467242915e-01 1.78349437979342689830630774849851e-01
            1.52715669637531192837087701263954e-02 6.77869789721086690725826429115841e-01 2.52933135839176426173935396946035e-01
            1.65119653090578903664731313938319e-02 5.96181882441888189028134092950495e-01 3.34657518177714385920751283265417e-01
            1.72593965508356095295283694213140e-02 5.09343076403983130440167315100553e-01 4.21087880063123087825971424535965e-01
            1.83813656226388989767261961105760e-02 6.87533646543294296726855918677757e-01 1.87462735345927888896966351239826e-01
            2.05056224747940211983188163458180e-02 6.09882363677961891745837874623248e-01 2.64906647850885523798325493771699e-01
            2.20521492709202793225387040365604e-02 5.25254911987722561406144450302236e-01 3.49006616007701775394878040970070e-01
            2.47835068399278185546918962245400e-02 5.33207332734684902320054789015558e-01 2.72179046638476918040794316766551e-01
            2.66731485051063289815065360244262e-02 4.46455240965651312112782989061088e-01 3.57596143740140481881439882272389e-01
            ];
        
        S2=[
            3.37656943579417217950883189914180e-04 3.52781468614782279732966507879155e-03
            2.59807342404100001867495173257794e-03 2.32544785996821011531121570214964e-02
            7.87684521611151100550785031373380e-03 6.13061711490491267895208693516906e-02
            1.15810108672388008266285197578327e-02 4.85597365316512985522479084465886e-01
            1.50579369917669004391713727386559e-02 1.19574379687203594113675819698983e-01
            2.19040063551890692994383869063313e-02 1.93832849735944889868477503114264e-01
            2.26289081705677505573781616021733e-02 4.36854825801192891976398868791875e-01
            2.84725810199424299307136010384056e-02 2.74786691854845999127832101294189e-01
            2.98211406539434994988901905799139e-02 3.61886974590005405794101989158662e-01
            ];
        
        S3=[];
        
        
    case  30
        
        % ALG. DEG.:   30
        % PTS CARD.:  190
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 2.9e-14
        
        S1=[
            7.51891383543372817180316936003237e-04 9.80030809579556794197685576364165e-01 1.60168116844335885196048963052817e-02
            1.15487000400619501359888996461223e-03 9.56210240585889215125803275441285e-01 4.00653234502971189456843603693414e-02
            1.99171703124402087875521871751516e-03 9.18991377915119755925843492150307e-01 7.64894022503812553992474931874312e-02
            2.76551913777887611867178918600985e-03 8.69266446632026346108546022151131e-01 1.25877991283982609838076882624591e-01
            3.28697960629342018565313132683059e-03 8.08161929939749557050276962399948e-01 1.86984740735145787970949982081947e-01
            3.47269493253384801262351189166111e-03 9.31015414728788681664184423425468e-01 4.79634591905694421254402470822242e-02
            3.65929391975805019809886431403356e-03 7.38140363468471782937285752268508e-01 2.56995801448648197862212327891029e-01
            3.96147782995318834681208741699265e-03 6.61964863274292047989888487791177e-01 3.33118225448962690382614937334438e-01
            4.20134868694696777396968556672618e-03 5.81189175108862587926239484659163e-01 4.13846657841551113232725356283481e-01
            5.52418246318765564228936426616201e-03 8.84274577566924291360805909789633e-01 9.11814742788630838710162151983241e-02
            6.93801881055061773345693509895682e-03 8.27639190276168834614622937806416e-01 1.46943421195845608240304613900662e-01
            7.85410394671830387747224477834607e-03 7.62587568566484019783047187956981e-01 2.12066839505368087692716017045313e-01
            8.64699211322754630881792792251872e-03 6.90118710560342352167140234087128e-01 2.84342556304586491222607946838252e-01
            9.10704875845551320434623221444781e-03 8.38318567454823804219188332353951e-01 1.01421462494414804655917805575882e-01
            9.35245872519548278323231471631516e-03 6.11809416665966332438131303206319e-01 3.62362307698883112649923532444518e-01
            9.81159064895887299273535830934634e-03 5.29148852224488042139682875131257e-01 4.44835853963697180901704086863901e-01
            1.09876697089070704538293909990898e-02 7.78589579815803345397284829232376e-01 1.59964054253387194526325743026973e-01
            1.25721314658824995236230392947618e-02 7.10623785618018422027830638398882e-01 2.27893984493791390555372800008627e-01
            1.38872646259526003748030476003805e-02 6.34277654844068061734674301987980e-01 3.03562715120035075955229331157170e-01
            1.47565688936917103341039947395075e-02 5.52626687887675771548856573645025e-01 3.84560072995415613839043089683400e-01
            1.49929194783670997892510712290459e-02 7.16871672741206555912185649503954e-01 1.71646103407863792611465214577038e-01
            1.70436028484623400269803994433460e-02 6.45570482899384456842994950420689e-01 2.42254408443608210133035640865273e-01
            1.85245809943621504711774861107187e-02 5.66843738230966565083690511528403e-01 3.19581218448555692823731533280807e-01
            1.93106125156711806722942270653220e-02 4.84400341724708916046893136808649e-01 4.01102153748808587518936974447570e-01
            2.11247953864554804948028277067351e-02 5.78477356973128076234047512116376e-01 2.45951501392227489528607975444174e-01
            2.31276136990037212148862977301178e-02 4.96282829301863182003273777809227e-01 3.25942627323928924010232321961666e-01
            2.64135805421229011291650579096313e-02 4.16568600024703472595177800030797e-01 3.31203743439010322990867507542134e-01
            ];
        
        S2=[
            2.08092049808074710300770382431779e-04 2.60849100043125713480063332383452e-03
            1.87714875195506801508482652707244e-03 1.99081215329428608262762878666763e-02
            4.29962879243079611202293932592511e-03 4.97508711341201725897320784497424e-01
            6.40688101660573730566916594852955e-03 5.38543920770780878970462879351544e-02
            1.25195516159525097737992282986852e-02 1.10234628175723006759945121757482e-01
            1.50685871206419092632744494153485e-02 4.68466155978488285605010332801612e-01
            1.82685494941227898058055956198587e-02 1.73692667715930587624129088908376e-01
            2.38585349675813108982769250587808e-02 4.10668319049822594379861584457103e-01
            2.45317376447564197694539700478344e-02 2.49556756982748401130223214750004e-01
            ];
        
        S3=[
            2.75546966950055705058986177391489e-02
            ];
        
        
    case  32
        
        % ALG. DEG.:   32
        % PTS CARD.:  210
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.1e-12
        
        S1=[
            7.30609041914017189912888827763027e-04 9.78394604898897179978689564450178e-01 1.78715560661563098443505026580169e-02
            7.44865669113147529749319186720413e-04 9.57376599458149901700210193666862e-01 3.94715791578275077000625969958492e-02
            1.41206581554954402162038729073856e-03 9.29763016779499351649462823843351e-01 6.62735149105613263742498020292260e-02
            2.22472763164065310409478470887734e-03 8.85614537945915492045401151699480e-01 1.10207799879050105662514624782489e-01
            2.78366500130545688196370690548065e-03 8.27604925832203464075575993774692e-01 1.68070571855654687931647117693501e-01
            2.82873288737946291387936881278620e-03 9.40702526955874196445961388235446e-01 4.04066851818969396958891593385488e-02
            3.10323315325580382714742988525813e-03 7.61456947235597203516022091207560e-01 2.34130448558700909877927642810391e-01
            3.27204429307987893632558851209069e-03 6.91098245011027567663575155165745e-01 3.04467727385771491643851049957448e-01
            3.49822807029196618591737610870496e-03 6.16960568881680049280191724392353e-01 3.78576515897704413138313839226612e-01
            3.71248336523432101441422048537788e-03 5.38216147161168834145428263582289e-01 4.57285169383451983726729395129951e-01
            4.41958002968257258197271042376997e-03 8.99401392370100527529075407073833e-01 7.94579312680107063826540070294868e-02
            5.83024885077109821657215249501860e-03 8.45458162390093193039319885428995e-01 1.32357395968752789006117609460489e-01
            6.80598314760665068501310770443524e-03 7.82095750516363930060492748452816e-01 1.95018153176870295828493340195564e-01
            7.24475644923155276400095203825913e-03 7.13726302839667536836998351645889e-01 2.63119135268292392293432158112410e-01
            7.36524644445563439476565292807209e-03 8.53648252342922631008548250974854e-01 9.44179770163390830184368951449869e-02
            7.60221467956296442780583788589865e-03 6.42663057493850664592116572748637e-01 3.34082493247255973845000198707567e-01
            8.17959909513327719510211721853921e-03 5.67421425923325672968644539650995e-01 4.09119706855425480096499768478679e-01
            9.22771056238034202701125963130835e-03 7.95883644812847901306440689950250e-01 1.49641910458703203401498171842832e-01
            1.04053751398834794933412339901224e-02 7.31688847720892487913602053595241e-01 2.12466086260680087072216792876134e-01
            1.12357477661229800819375768128339e-02 6.63321124311019305608283502806444e-01 2.80443055765805193679796047945274e-01
            1.22117109245319498816995107404182e-02 7.52105512553738697612004671100294e-01 1.48431512855652897320268834846502e-01
            1.22133940210877207921402742840655e-02 5.89842412472838084624982002424076e-01 3.53507535982596321488813373434823e-01
            1.29091240427459206380200740227338e-02 5.11476044318667555366175747622037e-01 4.31429040442822320233773325526272e-01
            1.44087823343037896989260460145488e-02 6.85135741196404968178512717713602e-01 2.13046127940301810443912700065994e-01
            1.61213754816722891483049551197837e-02 6.11188722458039679352737039152998e-01 2.86145793841655393663359063793905e-01
            1.73129623180881611621551030566479e-02 5.31047050460635050583846350491513e-01 3.65311627705974806268329757585889e-01
            1.77994991316269210446243675960432e-02 6.16755405364404762913466129248263e-01 2.23611895415582712098867546046677e-01
            1.97687866416255200729956698069145e-02 5.40939115423052063746922613063361e-01 2.97770116248729010965945462885429e-01
            2.09670196613528404816495509521701e-02 4.59837236521221992546770707122050e-01 3.77459830224483983140970622116583e-01
            2.30940410608222700117320158597067e-02 4.66649539182707573381492238695500e-01 3.03210043229811321285893654930987e-01
            ];
        
        S2=[
            2.86171847811123709025282746409857e-04 3.19145350613714295917588970041834e-03
            1.10999327125500509352851175037813e-03 1.64655689217531008006201176385730e-02
            5.69641734997258526035368575435314e-03 4.98495116354507120104422313033865e-02
            8.47073163470103175642922366250787e-03 4.88219324361224116248791915495531e-01
            9.28731828923172637868521661630439e-03 9.41148599404452168570855974394362e-02
            1.55097562817287093483997750809067e-02 1.56836966605033101274102591560222e-01
            1.77439996913240985443316333203256e-02 4.47917974916259398199258612294216e-01
            2.07895186780369806145429834032257e-02 2.27359360510421992795926371400128e-01
            2.39185100456866389651011672867753e-02 3.84316624257353078863985729185515e-01
            2.49866241525162902092915828689002e-02 3.06547093374997525749847682163818e-01
            ];
        
        S3=[];
        
        
        
    otherwise
        S1=[]; S2=[]; S3=[];
end




pointset_stats=[
    1    1 1.92e-16    0    0
    2    3 4.20e-16    0    0
    4    6 1.53e-16    0    0
    5   10 4.43e-16    0    0
    7   15 6.11e-16    0    0
    8   21 5.27e-16    0    0
    10   28 7.42e-16    0    0
    12   36 7.49e-16    0    0
    14   45 1.03e-15    0    0
    15   55 1.62e-15    0    0
    17   66 1.55e-15    0    0
    19   78 1.40e-15    0    0
    20   91 5.97e-16    0    0
    22  105 3.03e-14    0    0
    24  120 2.02e-14    0    0
    25  136 2.12e-14    0    0
    27  153 2.52e-13    0    0
    28  171 2.86e-15    0    0
    30  190 2.90e-14    0    0
    32  210 4.08e-12    0    0
    ];



