
function [xyw_bar,pointset_stats]=set_wandzura_xiao_barycentric(deg)

% --------------------------------------------------------------------
% CUBATURE RULE ON THE SIMPLEX [0 0; 0 1; 1 0] WITH ADE EQUAL TO deg.
% THE CUBATURE RULE IS DESCRIBED IN BARYCENTRICAL FORM.
% --------------------------------------------------------------------
% INPUT:
% deg: algebraic degree of precision (ADE) of the rule.
% --------------------------------------------------------------------
% OUTPUT:
% xyw_bar: MATRIX OF NODES xyw(:,1:3) IN BARYCENTRICAL COOR. AND WEIGHTS xyw(:,4).
%      REMARK: The sum of the weights makes 1.
%
% pointset_stats:
%    statistics of the pointset stored in a matrix. Each rows contains
%    degree, cardinality of the full pointset, moment error w.r.t. orthn. Dubiner basis,
%    number of points with negative weights, number of points outside the domain.
% --------------------------------------------------------------------
% SOURCE:
% S. Wandzura, H. Xiao,
% "Symmetric Quadrature Rules on a Triangle"
% Computers and Mathematics with Applications 45 (2003), pp.
% 1829-1840.
% --------------------------------------------------------------------
%----------------------------------------
% | DEG | CARD | W- | PT.O|   M.E.   |Q.|
%----------------------------------------
% |   5 |   7 |   0 |   0 | 2.50e-16 |PI|
% |  10 |  25 |   0 |   0 | 8.85e-16 |PI|
% |  15 |  54 |   0 |   0 | 1.39e-15 |PI|
% |  20 |  85 |   0 |   0 | 7.67e-16 |PI|
% |  25 | 126 |   0 |   0 | 9.16e-16 |PI|
% |  30 | 175 |   0 |   0 | 1.75e-15 |PI|
%---------------------------------------


%--------------------------------------------------------------------
% LEGEND:
%--------------------------------------------------------------------
% DEG  : algebraic degree of precision
% CARD : cardinality of the pointset
% W-   : negative weights
% PT.O : points outside the simplex [0 0; 1 0; 0 1].
% M.E. : maximum moment error in inf norm.
% Q.: quality of the rule.
%        PI: pos. weights, points in simplex
%        NI: some neg. weights, points in simplex
%        P0: pos. weights, some points out of the simplex
%        N0: some neg. weights, some points out of the simplex
%--------------------------------------------------------------------

switch deg
    
    case   5
        
        % ALG. DEG.:    5
        % PTS CARD.:    7
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 2.5e-16
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 2.25000000000000005551115123125783e-01
            4.70142064105115109473587153843255e-01 4.70142064105115109473587153843255e-01 5.97158717897697810528256923134904e-02 1.32394152788506219708963840275828e-01
            4.70142064105115109473587153843255e-01 5.97158717897697810528256923134904e-02 4.70142064105115109473587153843255e-01 1.32394152788506219708963840275828e-01
            5.97158717897697810528256923134904e-02 4.70142064105115109473587153843255e-01 4.70142064105115109473587153843255e-01 1.32394152788506219708963840275828e-01
            1.01286507323456370643910418039013e-01 1.01286507323456370643910418039013e-01 7.97426985353087314223330395179801e-01 1.25939180544827139529573400977824e-01
            1.01286507323456370643910418039013e-01 7.97426985353087314223330395179801e-01 1.01286507323456370643910418039013e-01 1.25939180544827139529573400977824e-01
            7.97426985353087314223330395179801e-01 1.01286507323456370643910418039013e-01 1.01286507323456370643910418039013e-01 1.25939180544827139529573400977824e-01
            ];
        
    case  10
        
        % ALG. DEG.:   10
        % PTS CARD.:   25
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 8.8e-16
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 8.35233998051963383302265242491558e-02
            4.97865432954474884663653710958897e-01 4.97865432954474884663653710958897e-01 4.26913409105023067269257808220573e-03 7.22985059205673991078455742353981e-03
            4.97865432954474884663653710958897e-01 4.26913409105023067269257808220573e-03 4.97865432954474884663653710958897e-01 7.22985059205673991078455742353981e-03
            4.26913409105023067269257808220573e-03 4.97865432954474884663653710958897e-01 4.97865432954474884663653710958897e-01 7.22985059205673991078455742353981e-03
            4.28012449729056221325862452431465e-01 4.28012449729056221325862452431465e-01 1.43975100541887557348275095137069e-01 7.44921779209804790777837979476317e-02
            4.28012449729056221325862452431465e-01 1.43975100541887557348275095137069e-01 4.28012449729056221325862452431465e-01 7.44921779209804790777837979476317e-02
            1.43975100541887557348275095137069e-01 4.28012449729056221325862452431465e-01 4.28012449729056221325862452431465e-01 7.44921779209804790777837979476317e-02
            1.84756412743224601724634226229682e-01 1.84756412743224601724634226229682e-01 6.30487174513550741039580316282809e-01 7.86464734031084855558901836047880e-02
            1.84756412743224601724634226229682e-01 6.30487174513550741039580316282809e-01 1.84756412743224601724634226229682e-01 7.86464734031084855558901836047880e-02
            6.30487174513550741039580316282809e-01 1.84756412743224601724634226229682e-01 1.84756412743224601724634226229682e-01 7.86464734031084855558901836047880e-02
            2.04812185716775514709997452200696e-02 2.04812185716775514709997452200696e-02 9.59037562856644876241318797838176e-01 6.92832308710750015279256075473313e-03
            2.04812185716775514709997452200696e-02 9.59037562856644876241318797838176e-01 2.04812185716775514709997452200696e-02 6.92832308710750015279256075473313e-03
            9.59037562856644876241318797838176e-01 2.04812185716775514709997452200696e-02 2.04812185716775514709997452200696e-02 6.92832308710750015279256075473313e-03
            1.36573576256033391018718248233199e-01 3.50029898972719771821360268404533e-02 8.28423433846694590165782301482977e-01 2.95183203347793854387504097758210e-02
            1.36573576256033391018718248233199e-01 8.28423433846694590165782301482977e-01 3.50029898972719771821360268404533e-02 2.95183203347793854387504097758210e-02
            3.50029898972719771821360268404533e-02 1.36573576256033391018718248233199e-01 8.28423433846694590165782301482977e-01 2.95183203347793854387504097758210e-02
            3.50029898972719771821360268404533e-02 8.28423433846694590165782301482977e-01 1.36573576256033391018718248233199e-01 2.95183203347793854387504097758210e-02
            8.28423433846694590165782301482977e-01 1.36573576256033391018718248233199e-01 3.50029898972719771821360268404533e-02 2.95183203347793854387504097758210e-02
            8.28423433846694590165782301482977e-01 3.50029898972719771821360268404533e-02 1.36573576256033391018718248233199e-01 2.95183203347793854387504097758210e-02
            3.32743600588638721582412927091355e-01 3.75490702584426669363715234339907e-02 6.29707329152918604542321645567426e-01 3.95793671960612200577678265744908e-02
            3.32743600588638721582412927091355e-01 6.29707329152918604542321645567426e-01 3.75490702584426669363715234339907e-02 3.95793671960612200577678265744908e-02
            3.75490702584426669363715234339907e-02 3.32743600588638721582412927091355e-01 6.29707329152918604542321645567426e-01 3.95793671960612200577678265744908e-02
            3.75490702584426669363715234339907e-02 6.29707329152918604542321645567426e-01 3.32743600588638721582412927091355e-01 3.95793671960612200577678265744908e-02
            6.29707329152918604542321645567426e-01 3.32743600588638721582412927091355e-01 3.75490702584426669363715234339907e-02 3.95793671960612200577678265744908e-02
            6.29707329152918604542321645567426e-01 3.75490702584426669363715234339907e-02 3.32743600588638721582412927091355e-01 3.95793671960612200577678265744908e-02
            ];
        
    case  15
        
        % ALG. DEG.:   15
        % PTS CARD.:   54
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.4e-15
        
        xyw_bar=[
            4.58280796369125109634978798567317e-01 4.58280796369125109634978798567317e-01 8.34384072617497807300424028653651e-02 3.26618188488053060036797603515879e-02
            4.58280796369125109634978798567317e-01 8.34384072617497807300424028653651e-02 4.58280796369125109634978798567317e-01 3.26618188488053060036797603515879e-02
            8.34384072617497807300424028653651e-02 4.58280796369125109634978798567317e-01 4.58280796369125109634978798567317e-01 3.26618188488053060036797603515879e-02
            4.03610464579130578943733098640223e-01 4.03610464579130578943733098640223e-01 1.92779070841738842112533802719554e-01 2.74128180313643748933216670593538e-02
            4.03610464579130578943733098640223e-01 1.92779070841738842112533802719554e-01 4.03610464579130578943733098640223e-01 2.74128180313643748933216670593538e-02
            1.92779070841738842112533802719554e-01 4.03610464579130578943733098640223e-01 4.03610464579130578943733098640223e-01 2.74128180313643748933216670593538e-02
            2.93197167913025413810146346804686e-01 2.93197167913025413810146346804686e-01 4.13605664173949172379707306390628e-01 2.65100365987033098624792160080688e-02
            2.93197167913025413810146346804686e-01 4.13605664173949172379707306390628e-01 2.93197167913025413810146346804686e-01 2.65100365987033098624792160080688e-02
            4.13605664173949172379707306390628e-01 2.93197167913025413810146346804686e-01 2.93197167913025413810146346804686e-01 2.65100365987033098624792160080688e-02
            1.46467786942772904001941469687154e-01 1.46467786942772904001941469687154e-01 7.07064426114454191996117060625693e-01 2.92159621364861225856213167162423e-02
            1.46467786942772904001941469687154e-01 7.07064426114454191996117060625693e-01 1.46467786942772904001941469687154e-01 2.92159621364861225856213167162423e-02
            7.07064426114454191996117060625693e-01 1.46467786942772904001941469687154e-01 1.46467786942772904001941469687154e-01 2.92159621364861225856213167162423e-02
            5.63628676656034027780961537246185e-02 5.63628676656034027780961537246185e-02 8.87274264668793222199383308179677e-01 1.05846080662439952224662675916989e-02
            5.63628676656034027780961537246185e-02 8.87274264668793222199383308179677e-01 5.63628676656034027780961537246185e-02 1.05846080662439952224662675916989e-02
            8.87274264668793222199383308179677e-01 5.63628676656034027780961537246185e-02 5.63628676656034027780961537246185e-02 1.05846080662439952224662675916989e-02
            1.65751268583703183223665433843053e-02 1.65751268583703183223665433843053e-02 9.66849746283259370294160817138618e-01 3.61464306409203664852136661522763e-03
            1.65751268583703183223665433843053e-02 9.66849746283259370294160817138618e-01 1.65751268583703183223665433843053e-02 3.61464306409203664852136661522763e-03
            9.66849746283259370294160817138618e-01 1.65751268583703183223665433843053e-02 1.65751268583703183223665433843053e-02 3.61464306409203664852136661522763e-03
            2.39534554154794410152362615917809e-01 9.91220330922478430613065114584970e-03 7.50553242535980769112313737423392e-01 8.52774810170943940723642384682535e-03
            2.39534554154794410152362615917809e-01 7.50553242535980769112313737423392e-01 9.91220330922478430613065114584970e-03 8.52774810170943940723642384682535e-03
            9.91220330922478430613065114584970e-03 2.39534554154794410152362615917809e-01 7.50553242535980769112313737423392e-01 8.52774810170943940723642384682535e-03
            9.91220330922478430613065114584970e-03 7.50553242535980769112313737423392e-01 2.39534554154794410152362615917809e-01 8.52774810170943940723642384682535e-03
            7.50553242535980769112313737423392e-01 2.39534554154794410152362615917809e-01 9.91220330922478430613065114584970e-03 8.52774810170943940723642384682535e-03
            7.50553242535980769112313737423392e-01 9.91220330922478430613065114584970e-03 2.39534554154794410152362615917809e-01 8.52774810170943940723642384682535e-03
            4.04878807318339972720622199631180e-01 1.58037706302280103742852190862322e-02 5.79317422051431996088410869560903e-01 1.39161765166919363517772367799807e-02
            4.04878807318339972720622199631180e-01 5.79317422051431996088410869560903e-01 1.58037706302280103742852190862322e-02 1.39161765166919363517772367799807e-02
            1.58037706302280103742852190862322e-02 4.04878807318339972720622199631180e-01 5.79317422051431996088410869560903e-01 1.39161765166919363517772367799807e-02
            1.58037706302280103742852190862322e-02 5.79317422051431996088410869560903e-01 4.04878807318339972720622199631180e-01 1.39161765166919363517772367799807e-02
            5.79317422051431996088410869560903e-01 4.04878807318339972720622199631180e-01 1.58037706302280103742852190862322e-02 1.39161765166919363517772367799807e-02
            5.79317422051431996088410869560903e-01 1.58037706302280103742852190862322e-02 4.04878807318339972720622199631180e-01 1.39161765166919363517772367799807e-02
            9.50021131130448365231799812136160e-02 5.14360881697066129464301909024471e-03 8.99854278069984525600943925383035e-01 4.29193294073483642275812499633503e-03
            9.50021131130448365231799812136160e-02 8.99854278069984525600943925383035e-01 5.14360881697066129464301909024471e-03 4.29193294073483642275812499633503e-03
            5.14360881697066129464301909024471e-03 9.50021131130448365231799812136160e-02 8.99854278069984525600943925383035e-01 4.29193294073483642275812499633503e-03
            5.14360881697066129464301909024471e-03 8.99854278069984525600943925383035e-01 9.50021131130448365231799812136160e-02 4.29193294073483642275812499633503e-03
            8.99854278069984525600943925383035e-01 9.50021131130448365231799812136160e-02 5.14360881697066129464301909024471e-03 4.29193294073483642275812499633503e-03
            8.99854278069984525600943925383035e-01 5.14360881697066129464301909024471e-03 9.50021131130448365231799812136160e-02 4.29193294073483642275812499633503e-03
            1.49753107322274009449714071706694e-01 4.89223257529887919914379779129376e-02 8.01324566924737191619954046473140e-01 1.62353292817748982856951300846049e-02
            1.49753107322274009449714071706694e-01 8.01324566924737191619954046473140e-01 4.89223257529887919914379779129376e-02 1.62353292817748982856951300846049e-02
            4.89223257529887919914379779129376e-02 1.49753107322274009449714071706694e-01 8.01324566924737191619954046473140e-01 1.62353292817748982856951300846049e-02
            4.89223257529887919914379779129376e-02 8.01324566924737191619954046473140e-01 1.49753107322274009449714071706694e-01 1.62353292817748982856951300846049e-02
            8.01324566924737191619954046473140e-01 1.49753107322274009449714071706694e-01 4.89223257529887919914379779129376e-02 1.62353292817748982856951300846049e-02
            8.01324566924737191619954046473140e-01 4.89223257529887919914379779129376e-02 1.49753107322274009449714071706694e-01 1.62353292817748982856951300846049e-02
            2.86919612441335014629828492616070e-01 6.87687486325192082192714337907091e-02 6.44311638926145735517536650149850e-01 2.56073409212623989694712633991003e-02
            2.86919612441335014629828492616070e-01 6.44311638926145735517536650149850e-01 6.87687486325192082192714337907091e-02 2.56073409212623989694712633991003e-02
            6.87687486325192082192714337907091e-02 2.86919612441335014629828492616070e-01 6.44311638926145735517536650149850e-01 2.56073409212623989694712633991003e-02
            6.87687486325192082192714337907091e-02 6.44311638926145735517536650149850e-01 2.86919612441335014629828492616070e-01 2.56073409212623989694712633991003e-02
            6.44311638926145735517536650149850e-01 2.86919612441335014629828492616070e-01 6.87687486325192082192714337907091e-02 2.56073409212623989694712633991003e-02
            6.44311638926145735517536650149850e-01 6.87687486325192082192714337907091e-02 2.86919612441335014629828492616070e-01 2.56073409212623989694712633991003e-02
            2.81835668099084613036353630377562e-01 1.68404418124699195580262767180102e-01 5.49759913776216246894534833700163e-01 3.30881955316456866089680488585145e-02
            2.81835668099084613036353630377562e-01 5.49759913776216246894534833700163e-01 1.68404418124699195580262767180102e-01 3.30881955316456866089680488585145e-02
            1.68404418124699195580262767180102e-01 2.81835668099084613036353630377562e-01 5.49759913776216246894534833700163e-01 3.30881955316456866089680488585145e-02
            1.68404418124699195580262767180102e-01 5.49759913776216246894534833700163e-01 2.81835668099084613036353630377562e-01 3.30881955316456866089680488585145e-02
            5.49759913776216246894534833700163e-01 2.81835668099084613036353630377562e-01 1.68404418124699195580262767180102e-01 3.30881955316456866089680488585145e-02
            5.49759913776216246894534833700163e-01 1.68404418124699195580262767180102e-01 2.81835668099084613036353630377562e-01 3.30881955316456866089680488585145e-02
            ];
        
    case  20
        
        % ALG. DEG.:   20
        % PTS CARD.:   85
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 7.7e-16
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 2.76104269976995206348213685032533e-02
            4.99249675337785525641720596468076e-01 4.99249675337785525641720596468076e-01 1.50064932442894871655880706384778e-03 1.77902954732674003363035453162411e-03
            4.99249675337785525641720596468076e-01 1.50064932442894871655880706384778e-03 4.99249675337785525641720596468076e-01 1.77902954732674003363035453162411e-03
            1.50064932442894871655880706384778e-03 4.99249675337785525641720596468076e-01 4.99249675337785525641720596468076e-01 1.77902954732674003363035453162411e-03
            4.52930124030524572376066316792276e-01 4.52930124030524572376066316792276e-01 9.41397519389508552478673664154485e-02 2.01123981139611704427316851706564e-02
            4.52930124030524572376066316792276e-01 9.41397519389508552478673664154485e-02 4.52930124030524572376066316792276e-01 2.01123981139611704427316851706564e-02
            9.41397519389508552478673664154485e-02 4.52930124030524572376066316792276e-01 4.52930124030524572376066316792276e-01 2.01123981139611704427316851706564e-02
            3.97763937955236790422475223749643e-01 3.97763937955236790422475223749643e-01 2.04472124089526419155049552500714e-01 2.68178472593315689320725425659475e-02
            3.97763937955236790422475223749643e-01 2.04472124089526419155049552500714e-01 3.97763937955236790422475223749643e-01 2.68178472593315689320725425659475e-02
            2.04472124089526419155049552500714e-01 3.97763937955236790422475223749643e-01 3.97763937955236790422475223749643e-01 2.68178472593315689320725425659475e-02
            2.64500202532787276332726378313964e-01 2.64500202532787276332726378313964e-01 4.70999594934425447334547243372072e-01 2.45231338015020096376961333817235e-02
            2.64500202532787276332726378313964e-01 4.70999594934425447334547243372072e-01 2.64500202532787276332726378313964e-01 2.45231338015020096376961333817235e-02
            4.70999594934425447334547243372072e-01 2.64500202532787276332726378313964e-01 2.64500202532787276332726378313964e-01 2.45231338015020096376961333817235e-02
            2.11018964092076793948038471171458e-01 2.11018964092076793948038471171458e-01 5.77962071815846467615074288914911e-01 1.63945784106953908187964685794213e-02
            2.11018964092076793948038471171458e-01 5.77962071815846467615074288914911e-01 2.11018964092076793948038471171458e-01 1.63945784106953908187964685794213e-02
            5.77962071815846467615074288914911e-01 2.11018964092076793948038471171458e-01 2.11018964092076793948038471171458e-01 1.63945784106953908187964685794213e-02
            1.07735607171271299820070055375254e-01 1.07735607171271299820070055375254e-01 7.84528785657457428115435504878405e-01 1.47959073986495999453172345283747e-02
            1.07735607171271299820070055375254e-01 7.84528785657457428115435504878405e-01 1.07735607171271299820070055375254e-01 1.47959073986495999453172345283747e-02
            7.84528785657457428115435504878405e-01 1.07735607171271299820070055375254e-01 1.07735607171271299820070055375254e-01 1.47959073986495999453172345283747e-02
            3.90690878378026629214048170979368e-02 3.90690878378026629214048170979368e-02 9.21861824324394674157190365804126e-01 4.57928227770425071152526896867130e-03
            3.90690878378026629214048170979368e-02 9.21861824324394674157190365804126e-01 3.90690878378026629214048170979368e-02 4.57928227770425071152526896867130e-03
            9.21861824324394674157190365804126e-01 3.90690878378026629214048170979368e-02 3.90690878378026629214048170979368e-02 4.57928227770425071152526896867130e-03
            1.11743797293295792777412245300184e-02 1.11743797293295792777412245300184e-02 9.77651240541340893486221830244176e-01 1.65182651557621703805600610337478e-03
            1.11743797293295792777412245300184e-02 9.77651240541340893486221830244176e-01 1.11743797293295792777412245300184e-02 1.65182651557621703805600610337478e-03
            9.77651240541340893486221830244176e-01 1.11743797293295792777412245300184e-02 1.11743797293295792777412245300184e-02 1.65182651557621703805600610337478e-03
            6.35496659083522041910896405170206e-02 5.34961818733725724683836233452894e-03 9.31100715904310538562071997148450e-01 2.34917090857558394367821108517091e-03
            6.35496659083522041910896405170206e-02 9.31100715904310538562071997148450e-01 5.34961818733725724683836233452894e-03 2.34917090857558394367821108517091e-03
            5.34961818733725724683836233452894e-03 6.35496659083522041910896405170206e-02 9.31100715904310538562071997148450e-01 2.34917090857558394367821108517091e-03
            5.34961818733725724683836233452894e-03 9.31100715904310538562071997148450e-01 6.35496659083522041910896405170206e-02 2.34917090857558394367821108517091e-03
            9.31100715904310538562071997148450e-01 6.35496659083522041910896405170206e-02 5.34961818733725724683836233452894e-03 2.34917090857558394367821108517091e-03
            9.31100715904310538562071997148450e-01 5.34961818733725724683836233452894e-03 6.35496659083522041910896405170206e-02 2.34917090857558394367821108517091e-03
            1.57106918940706996590250810186262e-01 7.95481706619893813581256836187094e-03 8.34938263993094009762785390194040e-01 4.46592575418179333485158011285421e-03
            1.57106918940706996590250810186262e-01 8.34938263993094009762785390194040e-01 7.95481706619893813581256836187094e-03 4.46592575418179333485158011285421e-03
            7.95481706619893813581256836187094e-03 1.57106918940706996590250810186262e-01 8.34938263993094009762785390194040e-01 4.46592575418179333485158011285421e-03
            7.95481706619893813581256836187094e-03 8.34938263993094009762785390194040e-01 1.57106918940706996590250810186262e-01 4.46592575418179333485158011285421e-03
            8.34938263993094009762785390194040e-01 1.57106918940706996590250810186262e-01 7.95481706619893813581256836187094e-03 4.46592575418179333485158011285421e-03
            8.34938263993094009762785390194040e-01 7.95481706619893813581256836187094e-03 1.57106918940706996590250810186262e-01 4.46592575418179333485158011285421e-03
            3.95642114364374075652364126653993e-01 1.04223982812637894213025901990477e-02 5.93935487354362190437484514404787e-01 6.09956680790797213520537667363897e-03
            3.95642114364374075652364126653993e-01 5.93935487354362190437484514404787e-01 1.04223982812637894213025901990477e-02 6.09956680790797213520537667363897e-03
            1.04223982812637894213025901990477e-02 3.95642114364374075652364126653993e-01 5.93935487354362190437484514404787e-01 6.09956680790797213520537667363897e-03
            1.04223982812637894213025901990477e-02 5.93935487354362190437484514404787e-01 3.95642114364374075652364126653993e-01 6.09956680790797213520537667363897e-03
            5.93935487354362190437484514404787e-01 3.95642114364374075652364126653993e-01 1.04223982812637894213025901990477e-02 6.09956680790797213520537667363897e-03
            5.93935487354362190437484514404787e-01 1.04223982812637894213025901990477e-02 3.95642114364374075652364126653993e-01 6.09956680790797213520537667363897e-03
            2.73167570712910479091561910536257e-01 1.09644147961233499877220509688414e-02 7.15868014490966264595783741242485e-01 6.89108132718820303591611420301888e-03
            2.73167570712910479091561910536257e-01 7.15868014490966264595783741242485e-01 1.09644147961233499877220509688414e-02 6.89108132718820303591611420301888e-03
            1.09644147961233499877220509688414e-02 2.73167570712910479091561910536257e-01 7.15868014490966264595783741242485e-01 6.89108132718820303591611420301888e-03
            1.09644147961233499877220509688414e-02 7.15868014490966264595783741242485e-01 2.73167570712910479091561910536257e-01 6.89108132718820303591611420301888e-03
            7.15868014490966264595783741242485e-01 2.73167570712910479091561910536257e-01 1.09644147961233499877220509688414e-02 6.89108132718820303591611420301888e-03
            7.15868014490966264595783741242485e-01 1.09644147961233499877220509688414e-02 2.73167570712910479091561910536257e-01 6.89108132718820303591611420301888e-03
            1.01785382485017103504354452070402e-01 3.85667120854623277081962839929474e-02 8.59647905429520520215191936586052e-01 7.99747507247816277753393876537302e-03
            1.01785382485017103504354452070402e-01 8.59647905429520520215191936586052e-01 3.85667120854623277081962839929474e-02 7.99747507247816277753393876537302e-03
            3.85667120854623277081962839929474e-02 1.01785382485017103504354452070402e-01 8.59647905429520520215191936586052e-01 7.99747507247816277753393876537302e-03
            3.85667120854623277081962839929474e-02 8.59647905429520520215191936586052e-01 1.01785382485017103504354452070402e-01 7.99747507247816277753393876537302e-03
            8.59647905429520520215191936586052e-01 1.01785382485017103504354452070402e-01 3.85667120854623277081962839929474e-02 7.99747507247816277753393876537302e-03
            8.59647905429520520215191936586052e-01 3.85667120854623277081962839929474e-02 1.01785382485017103504354452070402e-01 7.99747507247816277753393876537302e-03
            4.46658549176413788295292306429474e-01 3.55805078172181932227857714678976e-02 5.17760943006367990726346306473715e-01 7.38613428533602383446732275729119e-03
            4.46658549176413788295292306429474e-01 5.17760943006367990726346306473715e-01 3.55805078172181932227857714678976e-02 7.38613428533602383446732275729119e-03
            3.55805078172181932227857714678976e-02 4.46658549176413788295292306429474e-01 5.17760943006367990726346306473715e-01 7.38613428533602383446732275729119e-03
            3.55805078172181932227857714678976e-02 5.17760943006367990726346306473715e-01 4.46658549176413788295292306429474e-01 7.38613428533602383446732275729119e-03
            5.17760943006367990726346306473715e-01 4.46658549176413788295292306429474e-01 3.55805078172181932227857714678976e-02 7.38613428533602383446732275729119e-03
            5.17760943006367990726346306473715e-01 3.55805078172181932227857714678976e-02 4.46658549176413788295292306429474e-01 7.38613428533602383446732275729119e-03
            1.99010794149503095207265346289205e-01 4.96708163627641169357573858178512e-02 7.51318389487732773979189460078487e-01 1.27993318786482606491672342485799e-02
            1.99010794149503095207265346289205e-01 7.51318389487732773979189460078487e-01 4.96708163627641169357573858178512e-02 1.27993318786482606491672342485799e-02
            4.96708163627641169357573858178512e-02 1.99010794149503095207265346289205e-01 7.51318389487732773979189460078487e-01 1.27993318786482606491672342485799e-02
            4.96708163627641169357573858178512e-02 7.51318389487732773979189460078487e-01 1.99010794149503095207265346289205e-01 1.27993318786482606491672342485799e-02
            7.51318389487732773979189460078487e-01 1.99010794149503095207265346289205e-01 4.96708163627641169357573858178512e-02 1.27993318786482606491672342485799e-02
            7.51318389487732773979189460078487e-01 4.96708163627641169357573858178512e-02 1.99010794149503095207265346289205e-01 1.27993318786482606491672342485799e-02
            3.24261183692282684010166349253268e-01 5.85197250843317515189845323675399e-02 6.17219091223385585287530830100877e-01 1.72580711756965508862915470444932e-02
            3.24261183692282684010166349253268e-01 6.17219091223385585287530830100877e-01 5.85197250843317515189845323675399e-02 1.72580711756965508862915470444932e-02
            5.85197250843317515189845323675399e-02 3.24261183692282684010166349253268e-01 6.17219091223385585287530830100877e-01 1.72580711756965508862915470444932e-02
            5.85197250843317515189845323675399e-02 6.17219091223385585287530830100877e-01 3.24261183692282684010166349253268e-01 1.72580711756965508862915470444932e-02
            6.17219091223385585287530830100877e-01 3.24261183692282684010166349253268e-01 5.85197250843317515189845323675399e-02 1.72580711756965508862915470444932e-02
            6.17219091223385585287530830100877e-01 5.85197250843317515189845323675399e-02 3.24261183692282684010166349253268e-01 1.72580711756965508862915470444932e-02
            2.08531363210132908436733600865409e-01 1.21497787004394203203005986324570e-01 6.69970849785472943871411644067848e-01 1.86729459029354692700675144578781e-02
            2.08531363210132908436733600865409e-01 6.69970849785472943871411644067848e-01 1.21497787004394203203005986324570e-01 1.86729459029354692700675144578781e-02
            1.21497787004394203203005986324570e-01 2.08531363210132908436733600865409e-01 6.69970849785472943871411644067848e-01 1.86729459029354692700675144578781e-02
            1.21497787004394203203005986324570e-01 6.69970849785472943871411644067848e-01 2.08531363210132908436733600865409e-01 1.86729459029354692700675144578781e-02
            6.69970849785472943871411644067848e-01 2.08531363210132908436733600865409e-01 1.21497787004394203203005986324570e-01 1.86729459029354692700675144578781e-02
            6.69970849785472943871411644067848e-01 1.21497787004394203203005986324570e-01 2.08531363210132908436733600865409e-01 1.86729459029354692700675144578781e-02
            3.23170566536257475309668052432244e-01 1.40710844943938706563812957028858e-01 5.36118588519803873637670221796725e-01 2.28182240583952589130234400727204e-02
            3.23170566536257475309668052432244e-01 5.36118588519803873637670221796725e-01 1.40710844943938706563812957028858e-01 2.28182240583952589130234400727204e-02
            1.40710844943938706563812957028858e-01 3.23170566536257475309668052432244e-01 5.36118588519803873637670221796725e-01 2.28182240583952589130234400727204e-02
            1.40710844943938706563812957028858e-01 5.36118588519803873637670221796725e-01 3.23170566536257475309668052432244e-01 2.28182240583952589130234400727204e-02
            5.36118588519803873637670221796725e-01 3.23170566536257475309668052432244e-01 1.40710844943938706563812957028858e-01 2.28182240583952589130234400727204e-02
            5.36118588519803873637670221796725e-01 1.40710844943938706563812957028858e-01 3.23170566536257475309668052432244e-01 2.28182240583952589130234400727204e-02
            ];
        
    case  25
        
        % ALG. DEG.:   25
        % PTS CARD.:  126
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 9.2e-16
        
        xyw_bar=[
            4.86026758463412922317559150542365e-01 4.86026758463412922317559150542365e-01 2.79464830731741553648816989152692e-02 8.00558188002041190567403816658043e-03
            4.86026758463412922317559150542365e-01 2.79464830731741553648816989152692e-02 4.86026758463412922317559150542365e-01 8.00558188002041190567403816658043e-03
            2.79464830731741553648816989152692e-02 4.86026758463412922317559150542365e-01 4.86026758463412922317559150542365e-01 8.00558188002041190567403816658043e-03
            4.34410699336174321860681857288000e-01 4.34410699336174321860681857288000e-01 1.31178601327651356278636285424000e-01 1.59470768323904908692068005393594e-02
            4.34410699336174321860681857288000e-01 1.31178601327651356278636285424000e-01 4.34410699336174321860681857288000e-01 1.59470768323904908692068005393594e-02
            1.31178601327651356278636285424000e-01 4.34410699336174321860681857288000e-01 4.34410699336174321860681857288000e-01 1.59470768323904908692068005393594e-02
            3.89889135243963791399579577046097e-01 3.89889135243963791399579577046097e-01 2.20221729512072417200840845907805e-01 1.31091412307955210853060989961705e-02
            3.89889135243963791399579577046097e-01 2.20221729512072417200840845907805e-01 3.89889135243963791399579577046097e-01 1.31091412307955210853060989961705e-02
            2.20221729512072417200840845907805e-01 3.89889135243963791399579577046097e-01 3.89889135243963791399579577046097e-01 1.31091412307955210853060989961705e-02
            2.98443234019804382040064183456707e-01 2.98443234019804382040064183456707e-01 4.03113531960391235919871633086586e-01 1.95830009656356059111192990940253e-02
            2.98443234019804382040064183456707e-01 4.03113531960391235919871633086586e-01 2.98443234019804382040064183456707e-01 1.95830009656356059111192990940253e-02
            4.03113531960391235919871633086586e-01 2.98443234019804382040064183456707e-01 2.98443234019804382040064183456707e-01 1.95830009656356059111192990940253e-02
            2.34044172337371791936533327316283e-01 2.34044172337371791936533327316283e-01 5.31911655325256416126933345367434e-01 1.64708854415372592250310646022626e-02
            2.34044172337371791936533327316283e-01 5.31911655325256416126933345367434e-01 2.34044172337371791936533327316283e-01 1.64708854415372592250310646022626e-02
            5.31911655325256416126933345367434e-01 2.34044172337371791936533327316283e-01 2.34044172337371791936533327316283e-01 1.64708854415372592250310646022626e-02
            1.51468334609017596381619341627811e-01 1.51468334609017596381619341627811e-01 6.97063330781964807236761316744378e-01 8.54727907409209494704604992421082e-03
            1.51468334609017596381619341627811e-01 6.97063330781964807236761316744378e-01 1.51468334609017596381619341627811e-01 8.54727907409209494704604992421082e-03
            6.97063330781964807236761316744378e-01 1.51468334609017596381619341627811e-01 1.51468334609017596381619341627811e-01 8.54727907409209494704604992421082e-03
            1.12733893545993599238919102845102e-01 1.12733893545993599238919102845102e-01 7.74532212908012773766586178680882e-01 8.16188585722648664300127308024457e-03
            1.12733893545993599238919102845102e-01 7.74532212908012773766586178680882e-01 1.12733893545993599238919102845102e-01 8.16188585722648664300127308024457e-03
            7.74532212908012773766586178680882e-01 1.12733893545993599238919102845102e-01 1.12733893545993599238919102845102e-01 8.16188585722648664300127308024457e-03
            7.77156920915262616755825320069562e-02 7.77156920915262616755825320069562e-02 8.44568615816947421137683704728261e-01 6.12114653998377476057379809049053e-03
            7.77156920915262616755825320069562e-02 8.44568615816947421137683704728261e-01 7.77156920915262616755825320069562e-02 6.12114653998377476057379809049053e-03
            8.44568615816947421137683704728261e-01 7.77156920915262616755825320069562e-02 7.77156920915262616755825320069562e-02 6.12114653998377476057379809049053e-03
            3.48930936142970368751825560593716e-02 3.48930936142970368751825560593716e-02 9.30213812771405912371847080066800e-01 2.90849826493666313229069508849989e-03
            3.48930936142970368751825560593716e-02 9.30213812771405912371847080066800e-01 3.48930936142970368751825560593716e-02 2.90849826493666313229069508849989e-03
            9.30213812771405912371847080066800e-01 3.48930936142970368751825560593716e-02 3.48930936142970368751825560593716e-02 2.90849826493666313229069508849989e-03
            7.25818462093234025717114477060932e-03 7.25818462093234025717114477060932e-03 9.85483630758135276117570811038604e-01 6.92275245661995853978243342652377e-04
            7.25818462093234025717114477060932e-03 9.85483630758135276117570811038604e-01 7.25818462093234025717114477060932e-03 6.92275245661995853978243342652377e-04
            9.85483630758135276117570811038604e-01 7.25818462093234025717114477060932e-03 7.25818462093234025717114477060932e-03 6.92275245661995853978243342652377e-04
            2.27214452153364110387201435514726e-01 1.29235270444220495637821510825916e-03 7.71493195142193721736134648381267e-01 1.24828919927739609950501087354269e-03
            2.27214452153364110387201435514726e-01 7.71493195142193721736134648381267e-01 1.29235270444220495637821510825916e-03 1.24828919927739609950501087354269e-03
            1.29235270444220495637821510825916e-03 2.27214452153364110387201435514726e-01 7.71493195142193721736134648381267e-01 1.24828919927739609950501087354269e-03
            1.29235270444220495637821510825916e-03 7.71493195142193721736134648381267e-01 2.27214452153364110387201435514726e-01 1.24828919927739609950501087354269e-03
            7.71493195142193721736134648381267e-01 2.27214452153364110387201435514726e-01 1.29235270444220495637821510825916e-03 1.24828919927739609950501087354269e-03
            7.71493195142193721736134648381267e-01 1.29235270444220495637821510825916e-03 2.27214452153364110387201435514726e-01 1.24828919927739609950501087354269e-03
            4.35010554853571707045745142750093e-01 5.39970127211619656953933699128356e-03 5.59589743874312151028505013528047e-01 3.40475290880301985699252220740618e-03
            4.35010554853571707045745142750093e-01 5.59589743874312151028505013528047e-01 5.39970127211619656953933699128356e-03 3.40475290880301985699252220740618e-03
            5.39970127211619656953933699128356e-03 4.35010554853571707045745142750093e-01 5.59589743874312151028505013528047e-01 3.40475290880301985699252220740618e-03
            5.39970127211619656953933699128356e-03 5.59589743874312151028505013528047e-01 4.35010554853571707045745142750093e-01 3.40475290880301985699252220740618e-03
            5.59589743874312151028505013528047e-01 4.35010554853571707045745142750093e-01 5.39970127211619656953933699128356e-03 3.40475290880301985699252220740618e-03
            5.59589743874312151028505013528047e-01 5.39970127211619656953933699128356e-03 4.35010554853571707045745142750093e-01 3.40475290880301985699252220740618e-03
            3.20309599272204481579251478251535e-01 6.38400303397501558272386645853658e-03 6.73306397693820501970662917301524e-01 3.35965432606404870372052151594744e-03
            3.20309599272204481579251478251535e-01 6.73306397693820501970662917301524e-01 6.38400303397501558272386645853658e-03 3.35965432606404870372052151594744e-03
            6.38400303397501558272386645853658e-03 3.20309599272204481579251478251535e-01 6.73306397693820501970662917301524e-01 3.35965432606404870372052151594744e-03
            6.38400303397501558272386645853658e-03 6.73306397693820501970662917301524e-01 3.20309599272204481579251478251535e-01 3.35965432606404870372052151594744e-03
            6.73306397693820501970662917301524e-01 3.20309599272204481579251478251535e-01 6.38400303397501558272386645853658e-03 3.35965432606404870372052151594744e-03
            6.73306397693820501970662917301524e-01 6.38400303397501558272386645853658e-03 3.20309599272204481579251478251535e-01 3.35965432606404870372052151594744e-03
            9.17503222800051948526700584807259e-02 5.02821150199306288186562596820295e-03 9.03221466218001700632100892107701e-01 1.71615653949675297552934782885359e-03
            9.17503222800051948526700584807259e-02 9.03221466218001700632100892107701e-01 5.02821150199306288186562596820295e-03 1.71615653949675297552934782885359e-03
            5.02821150199306288186562596820295e-03 9.17503222800051948526700584807259e-02 9.03221466218001700632100892107701e-01 1.71615653949675297552934782885359e-03
            5.02821150199306288186562596820295e-03 9.03221466218001700632100892107701e-01 9.17503222800051948526700584807259e-02 1.71615653949675297552934782885359e-03
            9.03221466218001700632100892107701e-01 9.17503222800051948526700584807259e-02 5.02821150199306288186562596820295e-03 1.71615653949675297552934782885359e-03
            9.03221466218001700632100892107701e-01 5.02821150199306288186562596820295e-03 9.17503222800051948526700584807259e-02 1.71615653949675297552934782885359e-03
            3.80108358587243413961864746397623e-02 6.82675862178187386336558617472292e-03 9.55162405519493828975896576594096e-01 1.48085631671560492665651320720599e-03
            3.80108358587243413961864746397623e-02 9.55162405519493828975896576594096e-01 6.82675862178187386336558617472292e-03 1.48085631671560492665651320720599e-03
            6.82675862178187386336558617472292e-03 3.80108358587243413961864746397623e-02 9.55162405519493828975896576594096e-01 1.48085631671560492665651320720599e-03
            6.82675862178187386336558617472292e-03 9.55162405519493828975896576594096e-01 3.80108358587243413961864746397623e-02 1.48085631671560492665651320720599e-03
            9.55162405519493828975896576594096e-01 3.80108358587243413961864746397623e-02 6.82675862178187386336558617472292e-03 1.48085631671560492665651320720599e-03
            9.55162405519493828975896576594096e-01 6.82675862178187386336558617472292e-03 3.80108358587243413961864746397623e-02 1.48085631671560492665651320720599e-03
            1.57425218485311713134677802372607e-01 1.00161996399295398496587239378641e-02 8.32558581874758729668428713921458e-01 3.51131261072868284395243065887371e-03
            1.57425218485311713134677802372607e-01 8.32558581874758729668428713921458e-01 1.00161996399295398496587239378641e-02 3.51131261072868284395243065887371e-03
            1.00161996399295398496587239378641e-02 1.57425218485311713134677802372607e-01 8.32558581874758729668428713921458e-01 3.51131261072868284395243065887371e-03
            1.00161996399295398496587239378641e-02 8.32558581874758729668428713921458e-01 1.57425218485311713134677802372607e-01 3.51131261072868284395243065887371e-03
            8.32558581874758729668428713921458e-01 1.57425218485311713134677802372607e-01 1.00161996399295398496587239378641e-02 3.51131261072868284395243065887371e-03
            8.32558581874758729668428713921458e-01 1.00161996399295398496587239378641e-02 1.57425218485311713134677802372607e-01 3.51131261072868284395243065887371e-03
            2.39889659778533187450832997456018e-01 2.57578131733899907962470621214379e-02 7.34352527048076852977942508005071e-01 7.39355014970647858585373057849210e-03
            2.39889659778533187450832997456018e-01 7.34352527048076852977942508005071e-01 2.57578131733899907962470621214379e-02 7.39355014970647858585373057849210e-03
            2.57578131733899907962470621214379e-02 2.39889659778533187450832997456018e-01 7.34352527048076852977942508005071e-01 7.39355014970647858585373057849210e-03
            2.57578131733899907962470621214379e-02 7.34352527048076852977942508005071e-01 2.39889659778533187450832997456018e-01 7.39355014970647858585373057849210e-03
            7.34352527048076852977942508005071e-01 2.39889659778533187450832997456018e-01 2.57578131733899907962470621214379e-02 7.39355014970647858585373057849210e-03
            7.34352527048076852977942508005071e-01 2.57578131733899907962470621214379e-02 2.39889659778533187450832997456018e-01 7.39355014970647858585373057849210e-03
            3.61943118126060503492169573291903e-01 3.02278981199158094939072327633767e-02 6.07828983754023721708392713480862e-01 7.98308747737655303999826372773896e-03
            3.61943118126060503492169573291903e-01 6.07828983754023721708392713480862e-01 3.02278981199158094939072327633767e-02 7.98308747737655303999826372773896e-03
            3.02278981199158094939072327633767e-02 3.61943118126060503492169573291903e-01 6.07828983754023721708392713480862e-01 7.98308747737655303999826372773896e-03
            3.02278981199158094939072327633767e-02 6.07828983754023721708392713480862e-01 3.61943118126060503492169573291903e-01 7.98308747737655303999826372773896e-03
            6.07828983754023721708392713480862e-01 3.61943118126060503492169573291903e-01 3.02278981199158094939072327633767e-02 7.98308747737655303999826372773896e-03
            6.07828983754023721708392713480862e-01 3.02278981199158094939072327633767e-02 3.61943118126060503492169573291903e-01 7.98308747737655303999826372773896e-03
            8.35519609548285163702630029547436e-02 3.05049901071620804815509586660482e-02 8.85943048938009347637034807121381e-01 4.35596261315803878061414522449013e-03
            8.35519609548285163702630029547436e-02 8.85943048938009347637034807121381e-01 3.05049901071620804815509586660482e-02 4.35596261315803878061414522449013e-03
            3.05049901071620804815509586660482e-02 8.35519609548285163702630029547436e-02 8.85943048938009347637034807121381e-01 4.35596261315803878061414522449013e-03
            3.05049901071620804815509586660482e-02 8.85943048938009347637034807121381e-01 8.35519609548285163702630029547436e-02 4.35596261315803878061414522449013e-03
            8.85943048938009347637034807121381e-01 8.35519609548285163702630029547436e-02 3.05049901071620804815509586660482e-02 4.35596261315803878061414522449013e-03
            8.85943048938009347637034807121381e-01 3.05049901071620804815509586660482e-02 8.35519609548285163702630029547436e-02 4.35596261315803878061414522449013e-03
            1.48443220732418112994821512984345e-01 4.59565473625693429737815165481152e-02 8.05600231905012509336927450931398e-01 7.36505670141782663173923850763458e-03
            1.48443220732418112994821512984345e-01 8.05600231905012509336927450931398e-01 4.59565473625693429737815165481152e-02 7.36505670141782663173923850763458e-03
            4.59565473625693429737815165481152e-02 1.48443220732418112994821512984345e-01 8.05600231905012509336927450931398e-01 7.36505670141782663173923850763458e-03
            4.59565473625693429737815165481152e-02 8.05600231905012509336927450931398e-01 1.48443220732418112994821512984345e-01 7.36505670141782663173923850763458e-03
            8.05600231905012509336927450931398e-01 1.48443220732418112994821512984345e-01 4.59565473625693429737815165481152e-02 7.36505670141782663173923850763458e-03
            8.05600231905012509336927450931398e-01 4.59565473625693429737815165481152e-02 1.48443220732418112994821512984345e-01 7.36505670141782663173923850763458e-03
            2.83739708727535011600906500461861e-01 6.74428005402775559984362985233020e-02 6.48817490732187418522869393200381e-01 1.09635728464195433828898273986852e-02
            2.83739708727535011600906500461861e-01 6.48817490732187418522869393200381e-01 6.74428005402775559984362985233020e-02 1.09635728464195433828898273986852e-02
            6.74428005402775559984362985233020e-02 2.83739708727535011600906500461861e-01 6.48817490732187418522869393200381e-01 1.09635728464195433828898273986852e-02
            6.74428005402775559984362985233020e-02 6.48817490732187418522869393200381e-01 2.83739708727535011600906500461861e-01 1.09635728464195433828898273986852e-02
            6.48817490732187418522869393200381e-01 2.83739708727535011600906500461861e-01 6.74428005402775559984362985233020e-02 1.09635728464195433828898273986852e-02
            6.48817490732187418522869393200381e-01 6.74428005402775559984362985233020e-02 2.83739708727535011600906500461861e-01 1.09635728464195433828898273986852e-02
            4.06899375118787487437543859414291e-01 7.00450914159106013956446190604765e-02 5.23055533465301869533448098081863e-01 1.17499617435411120996979761343937e-02
            4.06899375118787487437543859414291e-01 5.23055533465301869533448098081863e-01 7.00450914159106013956446190604765e-02 1.17499617435411120996979761343937e-02
            7.00450914159106013956446190604765e-02 4.06899375118787487437543859414291e-01 5.23055533465301869533448098081863e-01 1.17499617435411120996979761343937e-02
            7.00450914159106013956446190604765e-02 5.23055533465301869533448098081863e-01 4.06899375118787487437543859414291e-01 1.17499617435411120996979761343937e-02
            5.23055533465301869533448098081863e-01 4.06899375118787487437543859414291e-01 7.00450914159106013956446190604765e-02 1.17499617435411120996979761343937e-02
            5.23055533465301869533448098081863e-01 7.00450914159106013956446190604765e-02 4.06899375118787487437543859414291e-01 1.17499617435411120996979761343937e-02
            1.94113987024892492572192281841126e-01 8.39115246401165998380022870151151e-02 7.21974488334990893712017623329302e-01 1.00156007137985631622045801236709e-02
            1.94113987024892492572192281841126e-01 7.21974488334990893712017623329302e-01 8.39115246401165998380022870151151e-02 1.00156007137985631622045801236709e-02
            8.39115246401165998380022870151151e-02 1.94113987024892492572192281841126e-01 7.21974488334990893712017623329302e-01 1.00156007137985631622045801236709e-02
            8.39115246401165998380022870151151e-02 7.21974488334990893712017623329302e-01 1.94113987024892492572192281841126e-01 1.00156007137985631622045801236709e-02
            7.21974488334990893712017623329302e-01 1.94113987024892492572192281841126e-01 8.39115246401165998380022870151151e-02 1.00156007137985631622045801236709e-02
            7.21974488334990893712017623329302e-01 8.39115246401165998380022870151151e-02 1.94113987024892492572192281841126e-01 1.00156007137985631622045801236709e-02
            3.24134347000703093577556046511745e-01 1.20375535677152700753111957965302e-01 5.55490117322144261180483226780780e-01 1.33096407876286716709390844926020e-02
            3.24134347000703093577556046511745e-01 5.55490117322144261180483226780780e-01 1.20375535677152700753111957965302e-01 1.33096407876286716709390844926020e-02
            1.20375535677152700753111957965302e-01 3.24134347000703093577556046511745e-01 5.55490117322144261180483226780780e-01 1.33096407876286716709390844926020e-02
            1.20375535677152700753111957965302e-01 5.55490117322144261180483226780780e-01 3.24134347000703093577556046511745e-01 1.33096407876286716709390844926020e-02
            5.55490117322144261180483226780780e-01 3.24134347000703093577556046511745e-01 1.20375535677152700753111957965302e-01 1.33096407876286716709390844926020e-02
            5.55490117322144261180483226780780e-01 1.20375535677152700753111957965302e-01 3.24134347000703093577556046511745e-01 1.33096407876286716709390844926020e-02
            2.29277483555980987617317623517010e-01 1.48066899157366704820049108093372e-01 6.22655617286652307562633268389618e-01 1.41544465052261311116543751609242e-02
            2.29277483555980987617317623517010e-01 6.22655617286652307562633268389618e-01 1.48066899157366704820049108093372e-01 1.41544465052261311116543751609242e-02
            1.48066899157366704820049108093372e-01 2.29277483555980987617317623517010e-01 6.22655617286652307562633268389618e-01 1.41544465052261311116543751609242e-02
            1.48066899157366704820049108093372e-01 6.22655617286652307562633268389618e-01 2.29277483555980987617317623517010e-01 1.41544465052261311116543751609242e-02
            6.22655617286652307562633268389618e-01 2.29277483555980987617317623517010e-01 1.48066899157366704820049108093372e-01 1.41544465052261311116543751609242e-02
            6.22655617286652307562633268389618e-01 1.48066899157366704820049108093372e-01 2.29277483555980987617317623517010e-01 1.41544465052261311116543751609242e-02
            3.25618122595983716660583695556852e-01 1.91771865867325086441042003571056e-01 4.82610011536691141387223069614265e-01 1.48813795611679993563525314925755e-02
            3.25618122595983716660583695556852e-01 4.82610011536691141387223069614265e-01 1.91771865867325086441042003571056e-01 1.48813795611679993563525314925755e-02
            1.91771865867325086441042003571056e-01 3.25618122595983716660583695556852e-01 4.82610011536691141387223069614265e-01 1.48813795611679993563525314925755e-02
            1.91771865867325086441042003571056e-01 4.82610011536691141387223069614265e-01 3.25618122595983716660583695556852e-01 1.48813795611679993563525314925755e-02
            4.82610011536691141387223069614265e-01 3.25618122595983716660583695556852e-01 1.91771865867325086441042003571056e-01 1.48813795611679993563525314925755e-02
            4.82610011536691141387223069614265e-01 1.91771865867325086441042003571056e-01 3.25618122595983716660583695556852e-01 1.48813795611679993563525314925755e-02
            ];
        
    case  30
        
        % ALG. DEG.:   30
        % PTS CARD.:  175
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.7e-15
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 1.55799602028992064534085315585799e-02
            4.96334941783617311639176250537275e-01 4.96334941783617311639176250537275e-01 7.33011643276537672164749892544933e-03 3.17723370053413526822860823983774e-03
            4.96334941783617311639176250537275e-01 7.33011643276537672164749892544933e-03 4.96334941783617311639176250537275e-01 3.17723370053413526822860823983774e-03
            7.33011643276537672164749892544933e-03 4.96334941783617311639176250537275e-01 4.96334941783617311639176250537275e-01 3.17723370053413526822860823983774e-03
            4.58502162098517684896137325267773e-01 4.58502162098517684896137325267773e-01 8.29956758029646302077253494644538e-02 1.04834266357307758638084393965073e-02
            4.58502162098517684896137325267773e-01 8.29956758029646302077253494644538e-02 4.58502162098517684896137325267773e-01 1.04834266357307758638084393965073e-02
            8.29956758029646302077253494644538e-02 4.58502162098517684896137325267773e-01 4.58502162098517684896137325267773e-01 1.04834266357307758638084393965073e-02
            4.24509521937294798110684723724262e-01 4.24509521937294798110684723724262e-01 1.50980956125410403778630552551476e-01 1.32094595777436357025846902502053e-02
            4.24509521937294798110684723724262e-01 1.50980956125410403778630552551476e-01 4.24509521937294798110684723724262e-01 1.32094595777436357025846902502053e-02
            1.50980956125410403778630552551476e-01 4.24509521937294798110684723724262e-01 4.24509521937294798110684723724262e-01 1.32094595777436357025846902502053e-02
            3.82047070053916693055384712351952e-01 3.82047070053916693055384712351952e-01 2.35905859892166613889230575296097e-01 1.49750069662715062923652453719114e-02
            3.82047070053916693055384712351952e-01 2.35905859892166613889230575296097e-01 3.82047070053916693055384712351952e-01 1.49750069662715062923652453719114e-02
            2.35905859892166613889230575296097e-01 3.82047070053916693055384712351952e-01 3.82047070053916693055384712351952e-01 1.49750069662715062923652453719114e-02
            2.80987845796075885740350486230454e-01 2.80987845796075885740350486230454e-01 4.38024308407848228519299027539091e-01 1.49879044433841972228105277054055e-02
            2.80987845796075885740350486230454e-01 4.38024308407848228519299027539091e-01 2.80987845796075885740350486230454e-01 1.49879044433841972228105277054055e-02
            4.38024308407848228519299027539091e-01 2.80987845796075885740350486230454e-01 2.80987845796075885740350486230454e-01 1.49879044433841972228105277054055e-02
            2.27348975854034407451109700559755e-01 2.27348975854034407451109700559755e-01 5.45302048291931185097780598880490e-01 1.33388647410216647920533361570961e-02
            2.27348975854034407451109700559755e-01 5.45302048291931185097780598880490e-01 2.27348975854034407451109700559755e-01 1.33388647410216647920533361570961e-02
            5.45302048291931185097780598880490e-01 2.27348975854034407451109700559755e-01 2.27348975854034407451109700559755e-01 1.33388647410216647920533361570961e-02
            1.74559111508729797312966525169031e-01 1.74559111508729797312966525169031e-01 6.50881776982540349862915718404111e-01 1.08891711139020158638501811765309e-02
            1.74559111508729797312966525169031e-01 6.50881776982540349862915718404111e-01 1.74559111508729797312966525169031e-01 1.08891711139020158638501811765309e-02
            6.50881776982540349862915718404111e-01 1.74559111508729797312966525169031e-01 1.74559111508729797312966525169031e-01 1.08891711139020158638501811765309e-02
            1.23258427201436593367844807289657e-01 1.23258427201436593367844807289657e-01 7.53483145597126813264310385420686e-01 8.18944066089346417447458748029021e-03
            1.23258427201436593367844807289657e-01 7.53483145597126813264310385420686e-01 1.23258427201436593367844807289657e-01 8.18944066089346417447458748029021e-03
            7.53483145597126813264310385420686e-01 1.23258427201436593367844807289657e-01 1.23258427201436593367844807289657e-01 8.18944066089346417447458748029021e-03
            8.00842288921968376014248747196689e-02 8.00842288921968376014248747196689e-02 8.39831542215606297041574634931749e-01 5.57538758860778773523980689219570e-03
            8.00842288921968376014248747196689e-02 8.39831542215606297041574634931749e-01 8.00842288921968376014248747196689e-02 5.57538758860778773523980689219570e-03
            8.39831542215606297041574634931749e-01 8.00842288921968376014248747196689e-02 8.00842288921968376014248747196689e-02 5.57538758860778773523980689219570e-03
            4.77744674078987782706029463497543e-02 4.77744674078987782706029463497543e-02 9.04451065184202485092157530743862e-01 3.19121647341197734598639002001619e-03
            4.77744674078987782706029463497543e-02 9.04451065184202485092157530743862e-01 4.77744674078987782706029463497543e-02 3.19121647341197734598639002001619e-03
            9.04451065184202485092157530743862e-01 4.77744674078987782706029463497543e-02 4.77744674078987782706029463497543e-02 3.19121647341197734598639002001619e-03
            2.17205146801415088786413321031432e-02 2.17205146801415088786413321031432e-02 9.56558970639716954487141720164800e-01 1.29671514432704569098897540158077e-03
            2.17205146801415088786413321031432e-02 9.56558970639716954487141720164800e-01 2.17205146801415088786413321031432e-02 1.29671514432704569098897540158077e-03
            9.56558970639716954487141720164800e-01 2.17205146801415088786413321031432e-02 2.17205146801415088786413321031432e-02 1.29671514432704569098897540158077e-03
            4.76467761543701399939987339848813e-03 4.76467761543701399939987339848813e-03 9.90470644769125940776177685620496e-01 2.98262826134917296075355075402058e-04
            4.76467761543701399939987339848813e-03 9.90470644769125940776177685620496e-01 4.76467761543701399939987339848813e-03 2.98262826134917296075355075402058e-04
            9.90470644769125940776177685620496e-01 4.76467761543701399939987339848813e-03 4.76467761543701399939987339848813e-03 2.98262826134917296075355075402058e-04
            4.15295270913311698190995002732961e-01 9.25371193346458120743136532837525e-04 5.83779357893341788177110629476374e-01 9.98905685078896841966322739381212e-04
            4.15295270913311698190995002732961e-01 5.83779357893341788177110629476374e-01 9.25371193346458120743136532837525e-04 9.98905685078896841966322739381212e-04
            9.25371193346458120743136532837525e-04 4.15295270913311698190995002732961e-01 5.83779357893341788177110629476374e-01 9.98905685078896841966322739381212e-04
            9.25371193346458120743136532837525e-04 5.83779357893341788177110629476374e-01 4.15295270913311698190995002732961e-01 9.98905685078896841966322739381212e-04
            5.83779357893341788177110629476374e-01 4.15295270913311698190995002732961e-01 9.25371193346458120743136532837525e-04 9.98905685078896841966322739381212e-04
            5.83779357893341788177110629476374e-01 9.25371193346458120743136532837525e-04 4.15295270913311698190995002732961e-01 9.98905685078896841966322739381212e-04
            6.11899097853490434029133382409782e-02 1.38592585556392600305841700247811e-03 9.37424164359087042086571273102891e-01 4.62850849173253526061544427250283e-04
            6.11899097853490434029133382409782e-02 9.37424164359087042086571273102891e-01 1.38592585556392600305841700247811e-03 4.62850849173253526061544427250283e-04
            1.38592585556392600305841700247811e-03 6.11899097853490434029133382409782e-02 9.37424164359087042086571273102891e-01 4.62850849173253526061544427250283e-04
            1.38592585556392600305841700247811e-03 9.37424164359087042086571273102891e-01 6.11899097853490434029133382409782e-02 4.62850849173253526061544427250283e-04
            9.37424164359087042086571273102891e-01 6.11899097853490434029133382409782e-02 1.38592585556392600305841700247811e-03 4.62850849173253526061544427250283e-04
            9.37424164359087042086571273102891e-01 1.38592585556392600305841700247811e-03 6.11899097853490434029133382409782e-02 4.62850849173253526061544427250283e-04
            1.64908690136906593837196055574168e-01 3.68241545591074803667197556933388e-03 8.31408894407182685881707584485412e-01 1.23445133638241358493836674625754e-03
            1.64908690136906593837196055574168e-01 8.31408894407182685881707584485412e-01 3.68241545591074803667197556933388e-03 1.23445133638241358493836674625754e-03
            3.68241545591074803667197556933388e-03 1.64908690136906593837196055574168e-01 8.31408894407182685881707584485412e-01 1.23445133638241358493836674625754e-03
            3.68241545591074803667197556933388e-03 8.31408894407182685881707584485412e-01 1.64908690136906593837196055574168e-01 1.23445133638241358493836674625754e-03
            8.31408894407182685881707584485412e-01 1.64908690136906593837196055574168e-01 3.68241545591074803667197556933388e-03 1.23445133638241358493836674625754e-03
            8.31408894407182685881707584485412e-01 3.68241545591074803667197556933388e-03 1.64908690136906593837196055574168e-01 1.23445133638241358493836674625754e-03
            2.50350622320024696065843272663187e-02 3.90322342415934907255969932293738e-03 9.71061714343838144891662977897795e-01 5.70719852243206367085892605217623e-04
            2.50350622320024696065843272663187e-02 9.71061714343838144891662977897795e-01 3.90322342415934907255969932293738e-03 5.70719852243206367085892605217623e-04
            3.90322342415934907255969932293738e-03 2.50350622320024696065843272663187e-02 9.71061714343838144891662977897795e-01 5.70719852243206367085892605217623e-04
            3.90322342415934907255969932293738e-03 9.71061714343838144891662977897795e-01 2.50350622320024696065843272663187e-02 5.70719852243206367085892605217623e-04
            9.71061714343838144891662977897795e-01 2.50350622320024696065843272663187e-02 3.90322342415934907255969932293738e-03 5.70719852243206367085892605217623e-04
            9.71061714343838144891662977897795e-01 3.90322342415934907255969932293738e-03 2.50350622320024696065843272663187e-02 5.70719852243206367085892605217623e-04
            3.06064465151095699280148210164043e-01 3.23324815501054496288957373906214e-03 6.90702286693893663382937120331917e-01 1.12694612587762449384953811204468e-03
            3.06064465151095699280148210164043e-01 6.90702286693893663382937120331917e-01 3.23324815501054496288957373906214e-03 1.12694612587762449384953811204468e-03
            3.23324815501054496288957373906214e-03 3.06064465151095699280148210164043e-01 6.90702286693893663382937120331917e-01 1.12694612587762449384953811204468e-03
            3.23324815501054496288957373906214e-03 6.90702286693893663382937120331917e-01 3.06064465151095699280148210164043e-01 1.12694612587762449384953811204468e-03
            6.90702286693893663382937120331917e-01 3.06064465151095699280148210164043e-01 3.23324815501054496288957373906214e-03 1.12694612587762449384953811204468e-03
            6.90702286693893663382937120331917e-01 3.23324815501054496288957373906214e-03 3.06064465151095699280148210164043e-01 1.12694612587762449384953811204468e-03
            1.07073283730218099529807318504027e-01 6.46743211223647484348076019955442e-03 8.86459284157545490678842270426685e-01 1.74786694940733794943754642048361e-03
            1.07073283730218099529807318504027e-01 8.86459284157545490678842270426685e-01 6.46743211223647484348076019955442e-03 1.74786694940733794943754642048361e-03
            6.46743211223647484348076019955442e-03 1.07073283730218099529807318504027e-01 8.86459284157545490678842270426685e-01 1.74786694940733794943754642048361e-03
            6.46743211223647484348076019955442e-03 8.86459284157545490678842270426685e-01 1.07073283730218099529807318504027e-01 1.74786694940733794943754642048361e-03
            8.86459284157545490678842270426685e-01 1.07073283730218099529807318504027e-01 6.46743211223647484348076019955442e-03 1.74786694940733794943754642048361e-03
            8.86459284157545490678842270426685e-01 6.46743211223647484348076019955442e-03 1.07073283730218099529807318504027e-01 1.74786694940733794943754642048361e-03
            2.29957549345584294497868427242793e-01 3.24747549133267820326009811537915e-03 7.66794975163083036839850592514267e-01 1.18281881503165737547222935432956e-03
            2.29957549345584294497868427242793e-01 7.66794975163083036839850592514267e-01 3.24747549133267820326009811537915e-03 1.18281881503165737547222935432956e-03
            3.24747549133267820326009811537915e-03 2.29957549345584294497868427242793e-01 7.66794975163083036839850592514267e-01 1.18281881503165737547222935432956e-03
            3.24747549133267820326009811537915e-03 7.66794975163083036839850592514267e-01 2.29957549345584294497868427242793e-01 1.18281881503165737547222935432956e-03
            7.66794975163083036839850592514267e-01 2.29957549345584294497868427242793e-01 3.24747549133267820326009811537915e-03 1.18281881503165737547222935432956e-03
            7.66794975163083036839850592514267e-01 3.24747549133267820326009811537915e-03 2.29957549345584294497868427242793e-01 1.18281881503165737547222935432956e-03
            3.37036633305782873382128173034289e-01 8.67509080675380774005933659509537e-03 6.54288275887463410818156717141392e-01 1.99083929467503469717959418971986e-03
            3.37036633305782873382128173034289e-01 6.54288275887463410818156717141392e-01 8.67509080675380774005933659509537e-03 1.99083929467503469717959418971986e-03
            8.67509080675380774005933659509537e-03 3.37036633305782873382128173034289e-01 6.54288275887463410818156717141392e-01 1.99083929467503469717959418971986e-03
            8.67509080675380774005933659509537e-03 6.54288275887463410818156717141392e-01 3.37036633305782873382128173034289e-01 1.99083929467503469717959418971986e-03
            6.54288275887463410818156717141392e-01 3.37036633305782873382128173034289e-01 8.67509080675380774005933659509537e-03 1.99083929467503469717959418971986e-03
            6.54288275887463410818156717141392e-01 8.67509080675380774005933659509537e-03 3.37036633305782873382128173034289e-01 1.99083929467503469717959418971986e-03
            5.62565761820607301846308700987720e-02 1.55970264673138698380272515464640e-02 9.28146397350625429467640969960485e-01 1.90041279503598082092907795015435e-03
            5.62565761820607301846308700987720e-02 9.28146397350625429467640969960485e-01 1.55970264673138698380272515464640e-02 1.90041279503598082092907795015435e-03
            1.55970264673138698380272515464640e-02 5.62565761820607301846308700987720e-02 9.28146397350625429467640969960485e-01 1.90041279503598082092907795015435e-03
            1.55970264673138698380272515464640e-02 9.28146397350625429467640969960485e-01 5.62565761820607301846308700987720e-02 1.90041279503598082092907795015435e-03
            9.28146397350625429467640969960485e-01 5.62565761820607301846308700987720e-02 1.55970264673138698380272515464640e-02 1.90041279503598082092907795015435e-03
            9.28146397350625429467640969960485e-01 1.55970264673138698380272515464640e-02 5.62565761820607301846308700987720e-02 1.90041279503598082092907795015435e-03
            4.02451375212400985770244687955710e-01 1.79767212536852093174299938027616e-02 5.79571903533913745931727135030087e-01 4.49836580881745294602236384662319e-03
            4.02451375212400985770244687955710e-01 5.79571903533913745931727135030087e-01 1.79767212536852093174299938027616e-02 4.49836580881745294602236384662319e-03
            1.79767212536852093174299938027616e-02 4.02451375212400985770244687955710e-01 5.79571903533913745931727135030087e-01 4.49836580881745294602236384662319e-03
            1.79767212536852093174299938027616e-02 5.79571903533913745931727135030087e-01 4.02451375212400985770244687955710e-01 4.49836580881745294602236384662319e-03
            5.79571903533913745931727135030087e-01 4.02451375212400985770244687955710e-01 1.79767212536852093174299938027616e-02 4.49836580881745294602236384662319e-03
            5.79571903533913745931727135030087e-01 1.79767212536852093174299938027616e-02 4.02451375212400985770244687955710e-01 4.49836580881745294602236384662319e-03
            2.43654702010828505542860966670560e-01 1.71242453538893411868482274940106e-02 7.39221052635282149800843853881815e-01 3.47871946027472058945995136980400e-03
            2.43654702010828505542860966670560e-01 7.39221052635282149800843853881815e-01 1.71242453538893411868482274940106e-02 3.47871946027472058945995136980400e-03
            1.71242453538893411868482274940106e-02 2.43654702010828505542860966670560e-01 7.39221052635282149800843853881815e-01 3.47871946027472058945995136980400e-03
            1.71242453538893411868482274940106e-02 7.39221052635282149800843853881815e-01 2.43654702010828505542860966670560e-01 3.47871946027472058945995136980400e-03
            7.39221052635282149800843853881815e-01 2.43654702010828505542860966670560e-01 1.71242453538893411868482274940106e-02 3.47871946027472058945995136980400e-03
            7.39221052635282149800843853881815e-01 1.71242453538893411868482274940106e-02 2.43654702010828505542860966670560e-01 3.47871946027472058945995136980400e-03
            1.65389585614532708124357895940193e-01 2.28834053465818802741704729442063e-02 8.11727009038885505276539333863184e-01 4.10239903672395510303427812459631e-03
            1.65389585614532708124357895940193e-01 8.11727009038885505276539333863184e-01 2.28834053465818802741704729442063e-02 4.10239903672395510303427812459631e-03
            2.28834053465818802741704729442063e-02 1.65389585614532708124357895940193e-01 8.11727009038885505276539333863184e-01 4.10239903672395510303427812459631e-03
            2.28834053465818802741704729442063e-02 8.11727009038885505276539333863184e-01 1.65389585614532708124357895940193e-01 4.10239903672395510303427812459631e-03
            8.11727009038885505276539333863184e-01 1.65389585614532708124357895940193e-01 2.28834053465818802741704729442063e-02 4.10239903672395510303427812459631e-03
            8.11727009038885505276539333863184e-01 2.28834053465818802741704729442063e-02 1.65389585614532708124357895940193e-01 4.10239903672395510303427812459631e-03
            9.93018744958468480010793655310408e-02 3.27375972877666718274802803989587e-02 8.67960528216386473232546450162772e-01 4.02176154974416381893131955393983e-03
            9.93018744958468480010793655310408e-02 8.67960528216386473232546450162772e-01 3.27375972877666718274802803989587e-02 4.02176154974416381893131955393983e-03
            3.27375972877666718274802803989587e-02 9.93018744958468480010793655310408e-02 8.67960528216386473232546450162772e-01 4.02176154974416381893131955393983e-03
            3.27375972877666718274802803989587e-02 8.67960528216386473232546450162772e-01 9.93018744958468480010793655310408e-02 4.02176154974416381893131955393983e-03
            8.67960528216386473232546450162772e-01 9.93018744958468480010793655310408e-02 3.27375972877666718274802803989587e-02 4.02176154974416381893131955393983e-03
            8.67960528216386473232546450162772e-01 3.27375972877666718274802803989587e-02 9.93018744958468480010793655310408e-02 4.02176154974416381893131955393983e-03
            3.08478333069055088433430000804947e-01 3.38210123423409170229092524095904e-02 6.57700654588604050054811978043290e-01 6.03316466079506855124670039458579e-03
            3.08478333069055088433430000804947e-01 6.57700654588604050054811978043290e-01 3.38210123423409170229092524095904e-02 6.03316466079506855124670039458579e-03
            3.38210123423409170229092524095904e-02 3.08478333069055088433430000804947e-01 6.57700654588604050054811978043290e-01 6.03316466079506855124670039458579e-03
            3.38210123423409170229092524095904e-02 6.57700654588604050054811978043290e-01 3.08478333069055088433430000804947e-01 6.03316466079506855124670039458579e-03
            6.57700654588604050054811978043290e-01 3.08478333069055088433430000804947e-01 3.38210123423409170229092524095904e-02 6.03316466079506855124670039458579e-03
            6.57700654588604050054811978043290e-01 3.38210123423409170229092524095904e-02 3.08478333069055088433430000804947e-01 6.03316466079506855124670039458579e-03
            4.60668318592112990828013607824687e-01 3.55476144600152671459980524559796e-02 5.03784066947871700392624916275963e-01 3.94629030212959985152165742761099e-03
            4.60668318592112990828013607824687e-01 5.03784066947871700392624916275963e-01 3.55476144600152671459980524559796e-02 3.94629030212959985152165742761099e-03
            3.55476144600152671459980524559796e-02 4.60668318592112990828013607824687e-01 5.03784066947871700392624916275963e-01 3.94629030212959985152165742761099e-03
            3.55476144600152671459980524559796e-02 5.03784066947871700392624916275963e-01 4.60668318592112990828013607824687e-01 3.94629030212959985152165742761099e-03
            5.03784066947871700392624916275963e-01 4.60668318592112990828013607824687e-01 3.55476144600152671459980524559796e-02 3.94629030212959985152165742761099e-03
            5.03784066947871700392624916275963e-01 3.55476144600152671459980524559796e-02 4.60668318592112990828013607824687e-01 3.94629030212959985152165742761099e-03
            2.18815299453929607009783353532839e-01 5.05397903068665402259718177901959e-02 7.30644910239203859703138732584193e-01 6.64404453768027095789117097979215e-03
            2.18815299453929607009783353532839e-01 7.30644910239203859703138732584193e-01 5.05397903068665402259718177901959e-02 6.64404453768027095789117097979215e-03
            5.05397903068665402259718177901959e-02 2.18815299453929607009783353532839e-01 7.30644910239203859703138732584193e-01 6.64404453768027095789117097979215e-03
            5.05397903068665402259718177901959e-02 7.30644910239203859703138732584193e-01 2.18815299453929607009783353532839e-01 6.64404453768027095789117097979215e-03
            7.30644910239203859703138732584193e-01 2.18815299453929607009783353532839e-01 5.05397903068665402259718177901959e-02 6.64404453768027095789117097979215e-03
            7.30644910239203859703138732584193e-01 5.05397903068665402259718177901959e-02 2.18815299453929607009783353532839e-01 6.64404453768027095789117097979215e-03
            3.79209551560274082948609475351986e-01 5.70147149157322133405934039274143e-02 5.63775733523993793916417871514568e-01 8.25430585607846156737110732137808e-03
            3.79209551560274082948609475351986e-01 5.63775733523993793916417871514568e-01 5.70147149157322133405934039274143e-02 8.25430585607846156737110732137808e-03
            5.70147149157322133405934039274143e-02 3.79209551560274082948609475351986e-01 5.63775733523993793916417871514568e-01 8.25430585607846156737110732137808e-03
            5.70147149157322133405934039274143e-02 5.63775733523993793916417871514568e-01 3.79209551560274082948609475351986e-01 8.25430585607846156737110732137808e-03
            5.63775733523993793916417871514568e-01 3.79209551560274082948609475351986e-01 5.70147149157322133405934039274143e-02 8.25430585607846156737110732137808e-03
            5.63775733523993793916417871514568e-01 5.70147149157322133405934039274143e-02 3.79209551560274082948609475351986e-01 8.25430585607846156737110732137808e-03
            1.42960819418185391960207653028192e-01 6.41528064212033982682825694610074e-02 7.92886374160611251404873200954171e-01 6.49605663340641330955049781437083e-03
            1.42960819418185391960207653028192e-01 7.92886374160611251404873200954171e-01 6.41528064212033982682825694610074e-02 6.49605663340641330955049781437083e-03
            6.41528064212033982682825694610074e-02 1.42960819418185391960207653028192e-01 7.92886374160611251404873200954171e-01 6.49605663340641330955049781437083e-03
            6.41528064212033982682825694610074e-02 7.92886374160611251404873200954171e-01 1.42960819418185391960207653028192e-01 6.49605663340641330955049781437083e-03
            7.92886374160611251404873200954171e-01 1.42960819418185391960207653028192e-01 6.41528064212033982682825694610074e-02 6.49605663340641330955049781437083e-03
            7.92886374160611251404873200954171e-01 6.41528064212033982682825694610074e-02 1.42960819418185391960207653028192e-01 6.49605663340641330955049781437083e-03
            2.83731282105925020076142573088873e-01 8.05011482876255984875157878377649e-02 6.35767569606449423069705062516732e-01 9.25277814414660580710769721690667e-03
            2.83731282105925020076142573088873e-01 6.35767569606449423069705062516732e-01 8.05011482876255984875157878377649e-02 9.25277814414660580710769721690667e-03
            8.05011482876255984875157878377649e-02 2.83731282105925020076142573088873e-01 6.35767569606449423069705062516732e-01 9.25277814414660580710769721690667e-03
            8.05011482876255984875157878377649e-02 6.35767569606449423069705062516732e-01 2.83731282105925020076142573088873e-01 9.25277814414660580710769721690667e-03
            6.35767569606449423069705062516732e-01 2.83731282105925020076142573088873e-01 8.05011482876255984875157878377649e-02 9.25277814414660580710769721690667e-03
            6.35767569606449423069705062516732e-01 8.05011482876255984875157878377649e-02 2.83731282105925020076142573088873e-01 9.25277814414660580710769721690667e-03
            1.96737441004440793967944500764133e-01 1.04367068134530505130186384121771e-01 6.98895490861028756413020346371923e-01 9.16492072629428340679869080531716e-03
            1.96737441004440793967944500764133e-01 6.98895490861028756413020346371923e-01 1.04367068134530505130186384121771e-01 9.16492072629428340679869080531716e-03
            1.04367068134530505130186384121771e-01 1.96737441004440793967944500764133e-01 6.98895490861028756413020346371923e-01 9.16492072629428340679869080531716e-03
            1.04367068134530505130186384121771e-01 6.98895490861028756413020346371923e-01 1.96737441004440793967944500764133e-01 9.16492072629428340679869080531716e-03
            6.98895490861028756413020346371923e-01 1.96737441004440793967944500764133e-01 1.04367068134530505130186384121771e-01 9.16492072629428340679869080531716e-03
            6.98895490861028756413020346371923e-01 1.04367068134530505130186384121771e-01 1.96737441004440793967944500764133e-01 9.16492072629428340679869080531716e-03
            3.55889141211662096697665447209147e-01 1.13844894428751300385904698941886e-01 5.30265964359586616794217661663424e-01 1.15695246280976757335956506267394e-02
            3.55889141211662096697665447209147e-01 5.30265964359586616794217661663424e-01 1.13844894428751300385904698941886e-01 1.15695246280976757335956506267394e-02
            1.13844894428751300385904698941886e-01 3.55889141211662096697665447209147e-01 5.30265964359586616794217661663424e-01 1.15695246280976757335956506267394e-02
            1.13844894428751300385904698941886e-01 5.30265964359586616794217661663424e-01 3.55889141211662096697665447209147e-01 1.15695246280976757335956506267394e-02
            5.30265964359586616794217661663424e-01 3.55889141211662096697665447209147e-01 1.13844894428751300385904698941886e-01 1.15695246280976757335956506267394e-02
            5.30265964359586616794217661663424e-01 1.13844894428751300385904698941886e-01 3.55889141211662096697665447209147e-01 1.15695246280976757335956506267394e-02
            2.59818685351911493608412229150417e-01 1.45363487715523709287168685477809e-01 5.94817826932564797104419085371774e-01 1.17611164676091756176035119096923e-02
            2.59818685351911493608412229150417e-01 5.94817826932564797104419085371774e-01 1.45363487715523709287168685477809e-01 1.17611164676091756176035119096923e-02
            1.45363487715523709287168685477809e-01 2.59818685351911493608412229150417e-01 5.94817826932564797104419085371774e-01 1.17611164676091756176035119096923e-02
            1.45363487715523709287168685477809e-01 5.94817826932564797104419085371774e-01 2.59818685351911493608412229150417e-01 1.17611164676091756176035119096923e-02
            5.94817826932564797104419085371774e-01 2.59818685351911493608412229150417e-01 1.45363487715523709287168685477809e-01 1.17611164676091756176035119096923e-02
            5.94817826932564797104419085371774e-01 1.45363487715523709287168685477809e-01 2.59818685351911493608412229150417e-01 1.17611164676091756176035119096923e-02
            3.21923181231298427551479335306794e-01 1.89945652821978699931193546035502e-01 4.88131165946722900272902734286617e-01 1.38247021821654072354368381070344e-02
            3.21923181231298427551479335306794e-01 4.88131165946722900272902734286617e-01 1.89945652821978699931193546035502e-01 1.38247021821654072354368381070344e-02
            1.89945652821978699931193546035502e-01 3.21923181231298427551479335306794e-01 4.88131165946722900272902734286617e-01 1.38247021821654072354368381070344e-02
            1.89945652821978699931193546035502e-01 4.88131165946722900272902734286617e-01 3.21923181231298427551479335306794e-01 1.38247021821654072354368381070344e-02
            4.88131165946722900272902734286617e-01 3.21923181231298427551479335306794e-01 1.89945652821978699931193546035502e-01 1.38247021821654072354368381070344e-02
            4.88131165946722900272902734286617e-01 1.89945652821978699931193546035502e-01 3.21923181231298427551479335306794e-01 1.38247021821654072354368381070344e-02
            ];
        
        
    otherwise
        xyw=[];
end




pointset_stats=[
    5    7 2.50e-16    0    0
    10   25 8.85e-16    0    0
    15   54 1.39e-15    0    0
    20   85 7.67e-16    0    0
    25  126 9.16e-16    0    0
    30  175 1.75e-15    0    0
    ];


