
function [xyw,pointset_stats]=set_witherden_vincent_standard(deg)

%--------------------------------------------------------------------
% CUBATURE RULE ON THE SIMPLEX [0 0; 0 1; 1 0] WITH ADE EQUAL TO deg.
% THE CUBATURE RULE IS DESCRIBED IN STANDARD FORM.
%--------------------------------------------------------------------
% INPUT:
% deg: algebraic degree of precision (ADE) of the rule.
%--------------------------------------------------------------------
% OUTPUT:
% xyw: MATRIX OF NODES xyw(:,1:2) AND WEIGHTS xyw(:,3).
%      REMARK: The sum of the weights makes 1/2.
%
% pointset_stats:
%    statistics of the pointset stored in a matrix. Each rows contains
%    degree, cardinality of the full pointset, moment error w.r.t. orthn.
%    Dubiner basis,number of points with negative weights, number of
%    points outside the domain.
%--------------------------------------------------------------------
% SOURCE:
%  F.D: Witherden, P.E. Vincent.
%  "On the identification of symmetric quadrature rules for finite
%  element methods".
%  Computers and Mathematics with Applications 69 (2015) 1232-1241.
%--------------------------------------------------------------------
% | DEG | CARD | W- | PT.O|   M.E.   |Q.|
%--------------------------------------------------------------------
% |   1 |   1 |   0 |   0 | 1.92e-16 |PI|
% |   4 |   6 |   0 |   0 | 1.39e-16 |PI|
% |   5 |   7 |   0 |   0 | 1.98e-16 |PI|
% |   6 |  12 |   0 |   0 | 2.36e-16 |PI|
% |   7 |  15 |   0 |   0 | 3.75e-16 |PI|
% |   8 |  16 |   0 |   0 | 8.05e-16 |PI|
% |   9 |  19 |   0 |   0 | 3.26e-16 |PI|
% |  10 |  25 |   0 |   0 | 4.86e-16 |PI|
% |  11 |  28 |   0 |   0 | 1.49e-15 |PI|
% |  12 |  33 |   0 |   0 | 6.52e-16 |PI|
% |  13 |  37 |   0 |   0 | 5.83e-16 |PI|
% |  14 |  42 |   0 |   0 | 4.37e-16 |PI|
% |  15 |  49 |   0 |   0 | 5.62e-16 |PI|
% |  16 |  55 |   0 |   0 | 6.11e-16 |PI|
% |  17 |  60 |   0 |   0 | 4.09e-16 |PI|
% |  18 |  67 |   0 |   0 | 9.00e-16 |PI|
% |  19 |  73 |   0 |   0 | 6.38e-16 |PI|
% |  20 |  79 |   0 |   0 | 7.46e-16 |PI|
%--------------------------------------------------------------------
% LEGEND:
%--------------------------------------------------------------------
% DEG  : algebraic degree of precision
% CARD : cardinality of the pointset
% W-   : negative weights
% PT.O : points outside the simplex [0 0; 1 0; 0 1].
% M.E. : maximum moment error in inf norm.
% Q.: quality of the rule.
%        PI: pos. weights, points in simplex
%        NI: some neg. weights, points in simplex
%        P0: pos. weights, some points out of the simplex
%        N0: some neg. weights, some points out of the simplex
%--------------------------------------------------------------------


switch deg
    
    case   1
        
        % ALG. DEG.:    1
        % PTS CARD.:    1
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.9e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 5.00000000000000000000000000000000e-01
            ];
        
    case   4
        
        % ALG. DEG.:    4
        % PTS CARD.:    6
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.4e-16
        
        xyw=[
            4.45948490915964890213274429697776e-01 4.45948490915964890213274429697776e-01 1.11690794839005735905601568447310e-01
            4.45948490915964890213274429697776e-01 1.08103018168070219573451140604448e-01 1.11690794839005735905601568447310e-01
            1.08103018168070219573451140604448e-01 4.45948490915964890213274429697776e-01 1.11690794839005735905601568447310e-01
            9.15762135097707430375635340169538e-02 9.15762135097707430375635340169538e-02 5.49758718276609353869943674908427e-02
            9.15762135097707430375635340169538e-02 8.16847572980458513924872931966092e-01 5.49758718276609353869943674908427e-02
            8.16847572980458513924872931966092e-01 9.15762135097707430375635340169538e-02 5.49758718276609353869943674908427e-02
            ];
        
    case   5
        
        % ALG. DEG.:    5
        % PTS CARD.:    7
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 2.0e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 1.12500000000000002775557561562891e-01
            1.01286507323456342888334802410100e-01 1.01286507323456342888334802410100e-01 6.29695902724135697647867004889122e-02
            1.01286507323456342888334802410100e-01 7.97426985353087314223330395179801e-01 6.29695902724135697647867004889122e-02
            7.97426985353087314223330395179801e-01 1.01286507323456342888334802410100e-01 6.29695902724135697647867004889122e-02
            4.70142064105115109473587153843255e-01 4.70142064105115109473587153843255e-01 6.61970763942530959766941123234574e-02
            4.70142064105115109473587153843255e-01 5.97158717897697810528256923134904e-02 6.61970763942530959766941123234574e-02
            5.97158717897697810528256923134904e-02 4.70142064105115109473587153843255e-01 6.61970763942530959766941123234574e-02
            ];
        
    case   6
        
        % ALG. DEG.:    6
        % PTS CARD.:   12
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 2.4e-16
        
        xyw=[
            6.30890144915022266225435032538371e-02 6.30890144915022266225435032538371e-02 2.54224531851034163398850296289311e-02
            6.30890144915022266225435032538371e-02 8.73821971016995546754912993492326e-01 2.54224531851034163398850296289311e-02
            8.73821971016995546754912993492326e-01 6.30890144915022266225435032538371e-02 2.54224531851034163398850296289311e-02
            2.49286745170910428726074314909056e-01 2.49286745170910428726074314909056e-01 5.83931378631896980113680228896555e-02
            2.49286745170910428726074314909056e-01 5.01426509658179142547851370181888e-01 5.83931378631896980113680228896555e-02
            5.01426509658179142547851370181888e-01 2.49286745170910428726074314909056e-01 5.83931378631896980113680228896555e-02
            5.31450498448169383891581674106419e-02 3.10352451033784393352732422499685e-01 4.14255378091867923484592495242396e-02
            5.31450498448169383891581674106419e-02 6.36502499121398668258109410089673e-01 4.14255378091867923484592495242396e-02
            3.10352451033784393352732422499685e-01 5.31450498448169383891581674106419e-02 4.14255378091867923484592495242396e-02
            3.10352451033784393352732422499685e-01 6.36502499121398668258109410089673e-01 4.14255378091867923484592495242396e-02
            6.36502499121398668258109410089673e-01 5.31450498448169383891581674106419e-02 4.14255378091867923484592495242396e-02
            6.36502499121398668258109410089673e-01 3.10352451033784393352732422499685e-01 4.14255378091867923484592495242396e-02
            ];
        
    case   7
        
        % ALG. DEG.:    7
        % PTS CARD.:   15
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.7e-16
        
        xyw=[
            3.37306485545878498299998682341538e-02 3.37306485545878498299998682341538e-02 8.27252505539606726447932061319079e-03
            3.37306485545878498299998682341538e-02 9.32538702890824300340000263531692e-01 8.27252505539606726447932061319079e-03
            9.32538702890824300340000263531692e-01 3.37306485545878498299998682341538e-02 8.27252505539606726447932061319079e-03
            2.41577382595403566956093754924950e-01 2.41577382595403566956093754924950e-01 6.39720856150778061088857384675066e-02
            2.41577382595403566956093754924950e-01 5.16845234809192866087812490150100e-01 6.39720856150778061088857384675066e-02
            5.16845234809192866087812490150100e-01 2.41577382595403566956093754924950e-01 6.39720856150778061088857384675066e-02
            4.74309692504718216632397798093734e-01 4.74309692504718216632397798093734e-01 3.85433230929930412123063376839127e-02
            4.74309692504718216632397798093734e-01 5.13806149905635667352044038125314e-02 3.85433230929930412123063376839127e-02
            5.13806149905635667352044038125314e-02 4.74309692504718216632397798093734e-01 3.85433230929930412123063376839127e-02
            4.70366446525952164314787751209224e-02 1.98683314797351573410821856668917e-01 2.79393664515998965680587673432456e-02
            4.70366446525952164314787751209224e-02 7.54280040550053265668850599467987e-01 2.79393664515998965680587673432456e-02
            1.98683314797351573410821856668917e-01 4.70366446525952164314787751209224e-02 2.79393664515998965680587673432456e-02
            1.98683314797351573410821856668917e-01 7.54280040550053265668850599467987e-01 2.79393664515998965680587673432456e-02
            7.54280040550053265668850599467987e-01 4.70366446525952164314787751209224e-02 2.79393664515998965680587673432456e-02
            7.54280040550053265668850599467987e-01 1.98683314797351573410821856668917e-01 2.79393664515998965680587673432456e-02
            ];
        
    case   8
        
        % ALG. DEG.:    8
        % PTS CARD.:   16
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 8.0e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 7.21578038388935860680817313550506e-02
            4.59292588292723125142913431773195e-01 4.59292588292723125142913431773195e-01 4.75458171336423096597911808203207e-02
            4.59292588292723125142913431773195e-01 8.14148234145537497141731364536099e-02 4.75458171336423096597911808203207e-02
            8.14148234145537497141731364536099e-02 4.59292588292723125142913431773195e-01 4.75458171336423096597911808203207e-02
            1.70569307751760212976677166807349e-01 1.70569307751760212976677166807349e-01 5.16086852673591223172877562319627e-02
            1.70569307751760212976677166807349e-01 6.58861384496479574046645666385302e-01 5.16086852673591223172877562319627e-02
            6.58861384496479574046645666385302e-01 1.70569307751760212976677166807349e-01 5.16086852673591223172877562319627e-02
            5.05472283170309566457945038564503e-02 5.05472283170309566457945038564503e-02 1.62292488115990396480015078850556e-02
            5.05472283170309566457945038564503e-02 8.98905543365938086708410992287099e-01 1.62292488115990396480015078850556e-02
            8.98905543365938086708410992287099e-01 5.05472283170309566457945038564503e-02 1.62292488115990396480015078850556e-02
            8.39477740995758781039626228448469e-03 2.63112829634638112352718053443823e-01 1.36151570872174963733325370185412e-02
            8.39477740995758781039626228448469e-03 7.28492392955404355348036915529519e-01 1.36151570872174963733325370185412e-02
            2.63112829634638112352718053443823e-01 8.39477740995758781039626228448469e-03 1.36151570872174963733325370185412e-02
            2.63112829634638112352718053443823e-01 7.28492392955404355348036915529519e-01 1.36151570872174963733325370185412e-02
            7.28492392955404355348036915529519e-01 8.39477740995758781039626228448469e-03 1.36151570872174963733325370185412e-02
            7.28492392955404355348036915529519e-01 2.63112829634638112352718053443823e-01 1.36151570872174963733325370185412e-02
            ];
        
    case   9
        
        % ALG. DEG.:    9
        % PTS CARD.:   19
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.3e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 4.85678981413994320659455183886166e-02
            4.37089591492936635486188379218220e-01 4.37089591492936635486188379218220e-01 3.89137705023871460774387287528953e-02
            4.37089591492936635486188379218220e-01 1.25820817014126729027623241563560e-01 3.89137705023871460774387287528953e-02
            1.25820817014126729027623241563560e-01 4.37089591492936635486188379218220e-01 3.89137705023871460774387287528953e-02
            1.88203535619032746861734040066949e-01 1.88203535619032746861734040066949e-01 3.98238694636051313024971420873044e-02
            1.88203535619032746861734040066949e-01 6.23592928761934506276531919866102e-01 3.98238694636051313024971420873044e-02
            6.23592928761934506276531919866102e-01 1.88203535619032746861734040066949e-01 3.98238694636051313024971420873044e-02
            4.89682519198737620236272505280795e-01 4.89682519198737620236272505280795e-01 1.56673501135695392161295558253187e-02
            4.89682519198737620236272505280795e-01 2.06349616025247595274549894384108e-02 1.56673501135695392161295558253187e-02
            2.06349616025247595274549894384108e-02 4.89682519198737620236272505280795e-01 1.56673501135695392161295558253187e-02
            4.47295133944526912550543329416541e-02 4.47295133944526912550543329416541e-02 1.27888378293490190956704921632081e-02
            4.47295133944526912550543329416541e-02 9.10540973211094617489891334116692e-01 1.27888378293490190956704921632081e-02
            9.10540973211094617489891334116692e-01 4.47295133944526912550543329416541e-02 1.27888378293490190956704921632081e-02
            3.68384120547362581277184290229343e-02 2.21962989160765677976172582930303e-01 2.16417696886446915549573333237277e-02
            3.68384120547362581277184290229343e-02 7.41198598784498008384957756788936e-01 2.16417696886446915549573333237277e-02
            2.21962989160765677976172582930303e-01 3.68384120547362581277184290229343e-02 2.16417696886446915549573333237277e-02
            2.21962989160765677976172582930303e-01 7.41198598784498008384957756788936e-01 2.16417696886446915549573333237277e-02
            7.41198598784498008384957756788936e-01 3.68384120547362581277184290229343e-02 2.16417696886446915549573333237277e-02
            7.41198598784498008384957756788936e-01 2.21962989160765677976172582930303e-01 2.16417696886446915549573333237277e-02
            ];
        
    case  10
        
        % ALG. DEG.:   10
        % PTS CARD.:   25
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.9e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 4.08716645731429864540906748970883e-02
            3.20553732169435168231075294897892e-02 3.20553732169435168231075294897892e-02 6.67648440657478327991958266807160e-03
            3.20553732169435168231075294897892e-02 9.35889253566112966353784941020422e-01 6.67648440657478327991958266807160e-03
            9.35889253566112966353784941020422e-01 3.20553732169435168231075294897892e-02 6.67648440657478327991958266807160e-03
            1.42161101056564376232671520483564e-01 1.42161101056564376232671520483564e-01 2.29789818023723654838441632364265e-02
            1.42161101056564376232671520483564e-01 7.15677797886871247534656959032873e-01 2.29789818023723654838441632364265e-02
            7.15677797886871247534656959032873e-01 1.42161101056564376232671520483564e-01 2.29789818023723654838441632364265e-02
            3.21812995288835446139330542791868e-01 5.30054118927343997924594987125602e-01 3.19524531982120219009146921962383e-02
            3.21812995288835446139330542791868e-01 1.48132885783820555936074470082531e-01 3.19524531982120219009146921962383e-02
            5.30054118927343997924594987125602e-01 3.21812995288835446139330542791868e-01 3.19524531982120219009146921962383e-02
            5.30054118927343997924594987125602e-01 1.48132885783820555936074470082531e-01 3.19524531982120219009146921962383e-02
            1.48132885783820555936074470082531e-01 3.21812995288835446139330542791868e-01 3.19524531982120219009146921962383e-02
            1.48132885783820555936074470082531e-01 5.30054118927343997924594987125602e-01 3.19524531982120219009146921962383e-02
            2.96198894887297892175581637275172e-02 6.01233328683459244956566180917434e-01 1.70923240814797143538594781375650e-02
            2.96198894887297892175581637275172e-02 3.69146781827810910314724424097221e-01 1.70923240814797143538594781375650e-02
            6.01233328683459244956566180917434e-01 2.96198894887297892175581637275172e-02 1.70923240814797143538594781375650e-02
            6.01233328683459244956566180917434e-01 3.69146781827810910314724424097221e-01 1.70923240814797143538594781375650e-02
            3.69146781827810910314724424097221e-01 2.96198894887297892175581637275172e-02 1.70923240814797143538594781375650e-02
            3.69146781827810910314724424097221e-01 6.01233328683459244956566180917434e-01 1.70923240814797143538594781375650e-02
            2.83676653399384526288429242413258e-02 1.63701733737182497652185020342586e-01 1.26488788536441923437969592214358e-02
            2.83676653399384526288429242413258e-02 8.07930600922879049718972055416089e-01 1.26488788536441923437969592214358e-02
            1.63701733737182497652185020342586e-01 2.83676653399384526288429242413258e-02 1.26488788536441923437969592214358e-02
            1.63701733737182497652185020342586e-01 8.07930600922879049718972055416089e-01 1.26488788536441923437969592214358e-02
            8.07930600922879049718972055416089e-01 2.83676653399384526288429242413258e-02 1.26488788536441923437969592214358e-02
            8.07930600922879049718972055416089e-01 1.63701733737182497652185020342586e-01 1.26488788536441923437969592214358e-02
            ];
        
    case  11
        
        % ALG. DEG.:   11
        % PTS CARD.:   28
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.5e-15
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 4.28805898661120885040354266948270e-02
            2.84854176143718995639630975347245e-02 2.84854176143718995639630975347245e-02 5.21593525644734566648619278339538e-03
            2.84854176143718995639630975347245e-02 9.43029164771256200872073804930551e-01 5.21593525644734566648619278339538e-03
            9.43029164771256200872073804930551e-01 2.84854176143718995639630975347245e-02 5.21593525644734566648619278339538e-03
            2.10219956703178278978327853110386e-01 2.10219956703178278978327853110386e-01 3.52578420558582739108288706120220e-02
            2.10219956703178278978327853110386e-01 5.79560086593643442043344293779228e-01 3.52578420558582739108288706120220e-02
            5.79560086593643442043344293779228e-01 2.10219956703178278978327853110386e-01 3.52578420558582739108288706120220e-02
            1.02635482712246428604885295499116e-01 1.02635482712246428604885295499116e-01 1.93153796185096537918468584393850e-02
            1.02635482712246428604885295499116e-01 7.94729034575507142790229409001768e-01 1.93153796185096537918468584393850e-02
            7.94729034575507142790229409001768e-01 1.02635482712246428604885295499116e-01 1.93153796185096537918468584393850e-02
            4.95891900965890919383838308931445e-01 4.95891900965890919383838308931445e-01 8.30313652729268089625058735236962e-03
            4.95891900965890919383838308931445e-01 8.21619806821816123232338213711046e-03 8.30313652729268089625058735236962e-03
            8.21619806821816123232338213711046e-03 4.95891900965890919383838308931445e-01 8.30313652729268089625058735236962e-03
            4.38465926764352198485852341036662e-01 4.38465926764352198485852341036662e-01 3.36580770397341341726438201931160e-02
            4.38465926764352198485852341036662e-01 1.23068146471295603028295317926677e-01 3.36580770397341341726438201931160e-02
            1.23068146471295603028295317926677e-01 4.38465926764352198485852341036662e-01 3.36580770397341341726438201931160e-02
            1.49324788652082374174057122218073e-01 8.43349783661853091842885987716727e-01 5.14514478647663635324516206992485e-03
            1.49324788652082374174057122218073e-01 7.32542768606458949420812132302672e-03 5.14514478647663635324516206992485e-03
            8.43349783661853091842885987716727e-01 1.49324788652082374174057122218073e-01 5.14514478647663635324516206992485e-03
            8.43349783661853091842885987716727e-01 7.32542768606458949420812132302672e-03 5.14514478647663635324516206992485e-03
            7.32542768606458949420812132302672e-03 1.49324788652082374174057122218073e-01 5.14514478647663635324516206992485e-03
            7.32542768606458949420812132302672e-03 8.43349783661853091842885987716727e-01 5.14514478647663635324516206992485e-03
            4.60105001654299572422246455971617e-02 2.89581125637705882880368335463572e-01 2.01662383202502668022582099638385e-02
            4.60105001654299572422246455971617e-02 6.64408374196864159877407018939266e-01 2.01662383202502668022582099638385e-02
            2.89581125637705882880368335463572e-01 4.60105001654299572422246455971617e-02 2.01662383202502668022582099638385e-02
            2.89581125637705882880368335463572e-01 6.64408374196864159877407018939266e-01 2.01662383202502668022582099638385e-02
            6.64408374196864159877407018939266e-01 4.60105001654299572422246455971617e-02 2.01662383202502668022582099638385e-02
            6.64408374196864159877407018939266e-01 2.89581125637705882880368335463572e-01 2.01662383202502668022582099638385e-02
            ];
        
    case  12
        
        % ALG. DEG.:   12
        % PTS CARD.:   33
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.5e-16
        
        xyw=[
            4.88203750945541525840809526926023e-01 4.88203750945541525840809526926023e-01 1.21334190407260122945842795161298e-02
            4.88203750945541525840809526926023e-01 2.35924981089169483183809461479541e-02 1.21334190407260122945842795161298e-02
            2.35924981089169483183809461479541e-02 4.88203750945541525840809526926023e-01 1.21334190407260122945842795161298e-02
            1.09257827659354267435531937735504e-01 1.09257827659354267435531937735504e-01 1.42430260344387684540157223977985e-02
            1.09257827659354267435531937735504e-01 7.81484344681291465128936124528991e-01 1.42430260344387684540157223977985e-02
            7.81484344681291465128936124528991e-01 1.09257827659354267435531937735504e-01 1.42430260344387684540157223977985e-02
            2.71462507014926079929040270144469e-01 2.71462507014926079929040270144469e-01 3.12706065979513753161356248710945e-02
            2.71462507014926079929040270144469e-01 4.57074985970147840141919459711062e-01 3.12706065979513753161356248710945e-02
            4.57074985970147840141919459711062e-01 2.71462507014926079929040270144469e-01 3.12706065979513753161356248710945e-02
            2.46463634363355832412878498871578e-02 2.46463634363355832412878498871578e-02 3.96582125498681856839988668639307e-03
            2.46463634363355832412878498871578e-02 9.50707273127328833517424300225684e-01 3.96582125498681856839988668639307e-03
            9.50707273127328833517424300225684e-01 2.46463634363355832412878498871578e-02 3.96582125498681856839988668639307e-03
            4.40111648658593090921442581020528e-01 4.40111648658593090921442581020528e-01 2.49591674640304642118771027980983e-02
            4.40111648658593090921442581020528e-01 1.19776702682813818157114837958943e-01 2.49591674640304642118771027980983e-02
            1.19776702682813818157114837958943e-01 4.40111648658593090921442581020528e-01 2.49591674640304642118771027980983e-02
            2.91655679738340944950891753251199e-01 6.85310163906391878185786481481045e-01 1.08917925193037778974769480555551e-02
            2.91655679738340944950891753251199e-01 2.30341563552671768633217652677558e-02 1.08917925193037778974769480555551e-02
            6.85310163906391878185786481481045e-01 2.91655679738340944950891753251199e-01 1.08917925193037778974769480555551e-02
            6.85310163906391878185786481481045e-01 2.30341563552671768633217652677558e-02 1.08917925193037778974769480555551e-02
            2.30341563552671768633217652677558e-02 2.91655679738340944950891753251199e-01 1.08917925193037778974769480555551e-02
            2.30341563552671768633217652677558e-02 6.85310163906391878185786481481045e-01 1.08917925193037778974769480555551e-02
            1.16296019677926576463278252049349e-01 2.55454228638517355509662820622907e-01 2.16136818297071008065923791718888e-02
            1.16296019677926576463278252049349e-01 6.28249751683556123538210158585571e-01 2.16136818297071008065923791718888e-02
            2.55454228638517355509662820622907e-01 1.16296019677926576463278252049349e-01 2.16136818297071008065923791718888e-02
            2.55454228638517355509662820622907e-01 6.28249751683556123538210158585571e-01 2.16136818297071008065923791718888e-02
            6.28249751683556123538210158585571e-01 1.16296019677926576463278252049349e-01 2.16136818297071008065923791718888e-02
            6.28249751683556123538210158585571e-01 2.55454228638517355509662820622907e-01 2.16136818297071008065923791718888e-02
            8.51337792510240110033237215247937e-01 1.27279717233589384495218155279872e-01 7.54183878825571713672148987939181e-03
            8.51337792510240110033237215247937e-01 2.13824902561705054715446294721914e-02 7.54183878825571713672148987939181e-03
            1.27279717233589384495218155279872e-01 8.51337792510240110033237215247937e-01 7.54183878825571713672148987939181e-03
            1.27279717233589384495218155279872e-01 2.13824902561705054715446294721914e-02 7.54183878825571713672148987939181e-03
            2.13824902561705054715446294721914e-02 8.51337792510240110033237215247937e-01 7.54183878825571713672148987939181e-03
            2.13824902561705054715446294721914e-02 1.27279717233589384495218155279872e-01 7.54183878825571713672148987939181e-03
            ];
        
    case  13
        
        % ALG. DEG.:   13
        % PTS CARD.:   37
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.8e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.39800182934158340186669988725043e-02
            4.89076946452539351728461269885884e-01 4.89076946452539351728461269885884e-01 1.19972009644473704553790938120983e-02
            4.89076946452539351728461269885884e-01 2.18461070949212965430774602282327e-02 1.19972009644473704553790938120983e-02
            2.18461070949212965430774602282327e-02 4.89076946452539351728461269885884e-01 1.19972009644473704553790938120983e-02
            2.21372286291832920390731942461571e-01 2.21372286291832920390731942461571e-01 2.91392425596000044507594139986395e-02
            2.21372286291832920390731942461571e-01 5.57255427416334159218536115076859e-01 2.91392425596000044507594139986395e-02
            5.57255427416334159218536115076859e-01 2.21372286291832920390731942461571e-01 2.91392425596000044507594139986395e-02
            4.26941414259800422481561099630198e-01 4.26941414259800422481561099630198e-01 2.78009837652266784957522105514727e-02
            4.26941414259800422481561099630198e-01 1.46117171480399155036877800739603e-01 2.78009837652266784957522105514727e-02
            1.46117171480399155036877800739603e-01 4.26941414259800422481561099630198e-01 2.78009837652266784957522105514727e-02
            2.15096811088431594249925637996057e-02 2.15096811088431594249925637996057e-02 3.02616855176958713877222173493919e-03
            2.15096811088431594249925637996057e-02 9.56980637782313681150014872400789e-01 3.02616855176958713877222173493919e-03
            9.56980637782313681150014872400789e-01 2.15096811088431594249925637996057e-02 3.02616855176958713877222173493919e-03
            8.78954830321972968576460516487714e-02 7.48507115899952224502555964136263e-01 1.20895199057969148642710521812660e-02
            8.78954830321972968576460516487714e-02 1.63597401067850478639797984214965e-01 1.20895199057969148642710521812660e-02
            7.48507115899952224502555964136263e-01 8.78954830321972968576460516487714e-02 1.20895199057969148642710521812660e-02
            7.48507115899952224502555964136263e-01 1.63597401067850478639797984214965e-01 1.20895199057969148642710521812660e-02
            1.63597401067850478639797984214965e-01 8.78954830321972968576460516487714e-02 1.20895199057969148642710521812660e-02
            1.63597401067850478639797984214965e-01 7.48507115899952224502555964136263e-01 1.20895199057969148642710521812660e-02
            1.10922042803463405391539708944038e-01 2.43701869010938265702748140029144e-02 7.48270055258283724175516482546300e-03
            1.10922042803463405391539708944038e-01 8.64707770295442768038185477053048e-01 7.48270055258283724175516482546300e-03
            2.43701869010938265702748140029144e-02 1.10922042803463405391539708944038e-01 7.48270055258283724175516482546300e-03
            2.43701869010938265702748140029144e-02 8.64707770295442768038185477053048e-01 7.48270055258283724175516482546300e-03
            8.64707770295442768038185477053048e-01 1.10922042803463405391539708944038e-01 7.48270055258283724175516482546300e-03
            8.64707770295442768038185477053048e-01 2.43701869010938265702748140029144e-02 7.48270055258283724175516482546300e-03
            3.08441760892117777803633771327441e-01 6.23545995553675513889402282075025e-01 1.73206380704241935664100537906052e-02
            3.08441760892117777803633771327441e-01 6.80122435542067638181151778553613e-02 1.73206380704241935664100537906052e-02
            6.23545995553675513889402282075025e-01 3.08441760892117777803633771327441e-01 1.73206380704241935664100537906052e-02
            6.23545995553675513889402282075025e-01 6.80122435542067638181151778553613e-02 1.73206380704241935664100537906052e-02
            6.80122435542067638181151778553613e-02 3.08441760892117777803633771327441e-01 1.73206380704241935664100537906052e-02
            6.80122435542067638181151778553613e-02 6.23545995553675513889402282075025e-01 1.73206380704241935664100537906052e-02
            5.12638910238238931782461804687046e-03 2.72515817773429702697285392787308e-01 4.79534050177163329031770899746334e-03
            5.12638910238238931782461804687046e-03 7.22357793124187907984889989165822e-01 4.79534050177163329031770899746334e-03
            2.72515817773429702697285392787308e-01 5.12638910238238931782461804687046e-03 4.79534050177163329031770899746334e-03
            2.72515817773429702697285392787308e-01 7.22357793124187907984889989165822e-01 4.79534050177163329031770899746334e-03
            7.22357793124187907984889989165822e-01 5.12638910238238931782461804687046e-03 4.79534050177163329031770899746334e-03
            7.22357793124187907984889989165822e-01 2.72515817773429702697285392787308e-01 4.79534050177163329031770899746334e-03
            ];
        
    case  14
        
        % ALG. DEG.:   14
        % PTS CARD.:   42
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.4e-16
        
        xyw=[
            1.77205532412543442788432912493590e-01 1.77205532412543442788432912493590e-01 2.10812943684965115043183914167457e-02
            1.77205532412543442788432912493590e-01 6.45588935174913114423134175012819e-01 2.10812943684965115043183914167457e-02
            6.45588935174913114423134175012819e-01 1.77205532412543442788432912493590e-01 2.10812943684965115043183914167457e-02
            4.17644719340453940414903399869218e-01 4.17644719340453940414903399869218e-01 1.63941767720626775661774132686332e-02
            4.17644719340453940414903399869218e-01 1.64710561319092119170193200261565e-01 1.63941767720626775661774132686332e-02
            1.64710561319092119170193200261565e-01 4.17644719340453940414903399869218e-01 1.63941767720626775661774132686332e-02
            6.17998830908725871324804757023230e-02 6.17998830908725871324804757023230e-02 7.21684983488833555614894166296835e-03
            6.17998830908725871324804757023230e-02 8.76400233818254825735039048595354e-01 7.21684983488833555614894166296835e-03
            8.76400233818254825735039048595354e-01 6.17998830908725871324804757023230e-02 7.21684983488833555614894166296835e-03
            4.88963910362178622026618768359185e-01 4.88963910362178622026618768359185e-01 1.09417906847144464493881699240774e-02
            4.88963910362178622026618768359185e-01 2.20721792756427559467624632816296e-02 1.09417906847144464493881699240774e-02
            2.20721792756427559467624632816296e-02 4.88963910362178622026618768359185e-01 1.09417906847144464493881699240774e-02
            2.73477528308838646609046918456443e-01 2.73477528308838646609046918456443e-01 2.58870522536457994822178818594693e-02
            2.73477528308838646609046918456443e-01 4.53044943382322706781906163087115e-01 2.58870522536457994822178818594693e-02
            4.53044943382322706781906163087115e-01 2.73477528308838646609046918456443e-01 2.58870522536457994822178818594693e-02
            1.93909612487010440950996326137101e-02 1.93909612487010440950996326137101e-02 2.46170180120004137430678170517240e-03
            1.93909612487010440950996326137101e-02 9.61218077502597911809800734772580e-01 2.46170180120004137430678170517240e-03
            9.61218077502597911809800734772580e-01 1.93909612487010440950996326137101e-02 2.46170180120004137430678170517240e-03
            2.98372882136257788765476561820833e-01 6.86980167808087793801519183034543e-01 7.21815405676692195546806374295556e-03
            2.98372882136257788765476561820833e-01 1.46469500556544174330042551446240e-02 7.21815405676692195546806374295556e-03
            6.86980167808087793801519183034543e-01 2.98372882136257788765476561820833e-01 7.21815405676692195546806374295556e-03
            6.86980167808087793801519183034543e-01 1.46469500556544174330042551446240e-02 7.21815405676692195546806374295556e-03
            1.46469500556544174330042551446240e-02 2.98372882136257788765476561820833e-01 7.21815405676692195546806374295556e-03
            1.46469500556544174330042551446240e-02 6.86980167808087793801519183034543e-01 7.21815405676692195546806374295556e-03
            5.71247574036479188741566304088337e-02 1.72266687821355568566161764465505e-01 1.23328766062818402649003957094465e-02
            5.71247574036479188741566304088337e-02 7.70608554774996457048530373867834e-01 1.23328766062818402649003957094465e-02
            1.72266687821355568566161764465505e-01 5.71247574036479188741566304088337e-02 1.23328766062818402649003957094465e-02
            1.72266687821355568566161764465505e-01 7.70608554774996457048530373867834e-01 1.23328766062818402649003957094465e-02
            7.70608554774996457048530373867834e-01 5.71247574036479188741566304088337e-02 1.23328766062818402649003957094465e-02
            7.70608554774996457048530373867834e-01 1.72266687821355568566161764465505e-01 1.23328766062818402649003957094465e-02
            3.36861459796344964168213209632086e-01 5.70222290846683188547672216373030e-01 1.92857553935303453751526348014522e-02
            3.36861459796344964168213209632086e-01 9.29162493569718472841145739948843e-02 1.92857553935303453751526348014522e-02
            5.70222290846683188547672216373030e-01 3.36861459796344964168213209632086e-01 1.92857553935303453751526348014522e-02
            5.70222290846683188547672216373030e-01 9.29162493569718472841145739948843e-02 1.92857553935303453751526348014522e-02
            9.29162493569718472841145739948843e-02 3.36861459796344964168213209632086e-01 1.92857553935303453751526348014522e-02
            9.29162493569718472841145739948843e-02 5.70222290846683188547672216373030e-01 1.92857553935303453751526348014522e-02
            1.26833093287204157206815580138937e-03 1.18974497696956837966553166552330e-01 2.50511441925033639596676060534719e-03
            1.26833093287204157206815580138937e-03 8.79757171370171064950227446388453e-01 2.50511441925033639596676060534719e-03
            1.18974497696956837966553166552330e-01 1.26833093287204157206815580138937e-03 2.50511441925033639596676060534719e-03
            1.18974497696956837966553166552330e-01 8.79757171370171064950227446388453e-01 2.50511441925033639596676060534719e-03
            8.79757171370171064950227446388453e-01 1.26833093287204157206815580138937e-03 2.50511441925033639596676060534719e-03
            8.79757171370171064950227446388453e-01 1.18974497696956837966553166552330e-01 2.50511441925033639596676060534719e-03
            ];
        
    case  15
        
        % ALG. DEG.:   15
        % PTS CARD.:   49
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.6e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 2.21676936910920330259244792614481e-02
            4.05362214133975495844453007521224e-01 4.05362214133975495844453007521224e-01 2.13568907857302793529274964612341e-02
            4.05362214133975495844453007521224e-01 1.89275571732049008311093984957552e-01 2.13568907857302793529274964612341e-02
            1.89275571732049008311093984957552e-01 4.05362214133975495844453007521224e-01 2.13568907857302793529274964612341e-02
            7.01735528999860025400892027391819e-02 7.01735528999860025400892027391819e-02 8.22236878131257960256039751811841e-03
            7.01735528999860025400892027391819e-02 8.59652894200027994919821594521636e-01 8.22236878131257960256039751811841e-03
            8.59652894200027994919821594521636e-01 7.01735528999860025400892027391819e-02 8.22236878131257960256039751811841e-03
            4.74170681438019769871061726007611e-01 4.74170681438019769871061726007611e-01 8.69807400038170516753943672938476e-03
            4.74170681438019769871061726007611e-01 5.16586371239604602578765479847789e-02 8.69807400038170516753943672938476e-03
            5.16586371239604602578765479847789e-02 4.74170681438019769871061726007611e-01 8.69807400038170516753943672938476e-03
            2.26378713420349597651437534295837e-01 2.26378713420349597651437534295837e-01 2.33916808643548114476917731963113e-02
            2.26378713420349597651437534295837e-01 5.47242573159300804697124931408325e-01 2.33916808643548114476917731963113e-02
            5.47242573159300804697124931408325e-01 2.26378713420349597651437534295837e-01 2.33916808643548114476917731963113e-02
            4.94996956769126195130326095750206e-01 4.94996956769126195130326095750206e-01 4.78692309123004196974582313828250e-03
            4.94996956769126195130326095750206e-01 1.00060864617476097393478084995877e-02 4.78692309123004196974582313828250e-03
            1.00060864617476097393478084995877e-02 4.94996956769126195130326095750206e-01 4.78692309123004196974582313828250e-03
            1.58117262509886447041651535982965e-02 1.58117262509886447041651535982965e-02 1.48038731895268728736003716051073e-03
            1.58117262509886447041651535982965e-02 9.68376547498022710591669692803407e-01 1.48038731895268728736003716051073e-03
            9.68376547498022710591669692803407e-01 1.58117262509886447041651535982965e-02 1.48038731895268728736003716051073e-03
            1.83761123856810981713749697519233e-02 6.66975644801868106092967991571641e-01 7.80128641528798037751979421727810e-03
            1.83761123856810981713749697519233e-02 3.14648242812450740224505807418609e-01 7.80128641528798037751979421727810e-03
            6.66975644801868106092967991571641e-01 1.83761123856810981713749697519233e-02 7.80128641528798037751979421727810e-03
            6.66975644801868106092967991571641e-01 3.14648242812450740224505807418609e-01 7.80128641528798037751979421727810e-03
            3.14648242812450740224505807418609e-01 1.83761123856810981713749697519233e-02 7.80128641528798037751979421727810e-03
            3.14648242812450740224505807418609e-01 6.66975644801868106092967991571641e-01 7.80128641528798037751979421727810e-03
            9.13923703730840086834064095455687e-03 7.09486052364555197513595885538962e-02 2.01492668600904926284878015962931e-03
            9.13923703730840086834064095455687e-03 9.19912157726236023869148539233720e-01 2.01492668600904926284878015962931e-03
            7.09486052364555197513595885538962e-02 9.13923703730840086834064095455687e-03 2.01492668600904926284878015962931e-03
            7.09486052364555197513595885538962e-02 9.19912157726236023869148539233720e-01 2.01492668600904926284878015962931e-03
            9.19912157726236023869148539233720e-01 9.13923703730840086834064095455687e-03 2.01492668600904926284878015962931e-03
            9.19912157726236023869148539233720e-01 7.09486052364555197513595885538962e-02 2.01492668600904926284878015962931e-03
            1.90535589476393929508901692315703e-01 7.15222356931450642392178451700602e-01 1.43602934626006675106202337133254e-02
            1.90535589476393929508901692315703e-01 9.42420535921554280989198559836950e-02 1.43602934626006675106202337133254e-02
            7.15222356931450642392178451700602e-01 1.90535589476393929508901692315703e-01 1.43602934626006675106202337133254e-02
            7.15222356931450642392178451700602e-01 9.42420535921554280989198559836950e-02 1.43602934626006675106202337133254e-02
            9.42420535921554280989198559836950e-02 1.90535589476393929508901692315703e-01 1.43602934626006675106202337133254e-02
            9.42420535921554280989198559836950e-02 7.15222356931450642392178451700602e-01 1.43602934626006675106202337133254e-02
            1.68068645222414381201758715178585e-01 1.86387137281663894938787962018978e-02 5.83631059078792199107388327661283e-03
            1.68068645222414381201758715178585e-01 8.13292641049419229304362488619518e-01 5.83631059078792199107388327661283e-03
            1.86387137281663894938787962018978e-02 1.68068645222414381201758715178585e-01 5.83631059078792199107388327661283e-03
            1.86387137281663894938787962018978e-02 8.13292641049419229304362488619518e-01 5.83631059078792199107388327661283e-03
            8.13292641049419229304362488619518e-01 1.68068645222414381201758715178585e-01 5.83631059078792199107388327661283e-03
            8.13292641049419229304362488619518e-01 1.86387137281663894938787962018978e-02 5.83631059078792199107388327661283e-03
            3.38950611475277163719965756172314e-01 5.65252664877114230357335600274382e-01 1.56577381424846395763239570442238e-02
            3.38950611475277163719965756172314e-01 9.57967236476086059226986435533036e-02 1.56577381424846395763239570442238e-02
            5.65252664877114230357335600274382e-01 3.38950611475277163719965756172314e-01 1.56577381424846395763239570442238e-02
            5.65252664877114230357335600274382e-01 9.57967236476086059226986435533036e-02 1.56577381424846395763239570442238e-02
            9.57967236476086059226986435533036e-02 3.38950611475277163719965756172314e-01 1.56577381424846395763239570442238e-02
            9.57967236476086059226986435533036e-02 5.65252664877114230357335600274382e-01 1.56577381424846395763239570442238e-02
            ];
        
    case  16
        
        % ALG. DEG.:   16
        % PTS CARD.:   55
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.1e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 2.26322830369093917768363866116488e-02
            2.45990070467141719312564873689553e-01 2.45990070467141719312564873689553e-01 2.05464615718494725271625611640047e-02
            2.45990070467141719312564873689553e-01 5.08019859065716561374870252620894e-01 2.05464615718494725271625611640047e-02
            5.08019859065716561374870252620894e-01 2.45990070467141719312564873689553e-01 2.05464615718494725271625611640047e-02
            4.15584896885420496115415289750672e-01 4.15584896885420496115415289750672e-01 2.03559166562126761523821016908187e-02
            4.15584896885420496115415289750672e-01 1.68830206229159007769169420498656e-01 2.03559166562126761523821016908187e-02
            1.68830206229159007769169420498656e-01 4.15584896885420496115415289750672e-01 2.03559166562126761523821016908187e-02
            8.53555665867003487967679120629327e-02 8.53555665867003487967679120629327e-02 7.39081734511220014849364190467895e-03
            8.53555665867003487967679120629327e-02 8.29288866826599302406464175874135e-01 7.39081734511220014849364190467895e-03
            8.29288866826599302406464175874135e-01 8.53555665867003487967679120629327e-02 7.39081734511220014849364190467895e-03
            1.61918644191271221544070613163058e-01 1.61918644191271221544070613163058e-01 1.47092048494940463160274290999041e-02
            1.61918644191271221544070613163058e-01 6.76162711617457556911858773673885e-01 1.47092048494940463160274290999041e-02
            6.76162711617457556911858773673885e-01 1.61918644191271221544070613163058e-01 1.47092048494940463160274290999041e-02
            5.00000000000000000000000000000000e-01 5.00000000000000000000000000000000e-01 2.20927315607528408636373029594324e-03
            5.00000000000000000000000000000000e-01 0.00000000000000000000000000000000e+00 2.20927315607528408636373029594324e-03
            0.00000000000000000000000000000000e+00 5.00000000000000000000000000000000e-01 2.20927315607528408636373029594324e-03
            4.75280727545942083267505040566903e-01 4.75280727545942083267505040566903e-01 1.29871666491385758662735483426331e-02
            4.75280727545942083267505040566903e-01 4.94385449081158334649899188661948e-02 1.29871666491385758662735483426331e-02
            4.94385449081158334649899188661948e-02 4.75280727545942083267505040566903e-01 1.29871666491385758662735483426331e-02
            5.47551749147031086017989309766563e-02 7.54170061444767725333804264664650e-01 9.46913623220784796130633509392283e-03
            5.47551749147031086017989309766563e-02 1.91074763640529221575548035616521e-01 9.46913623220784796130633509392283e-03
            7.54170061444767725333804264664650e-01 5.47551749147031086017989309766563e-02 9.46913623220784796130633509392283e-03
            7.54170061444767725333804264664650e-01 1.91074763640529221575548035616521e-01 9.46913623220784796130633509392283e-03
            1.91074763640529221575548035616521e-01 5.47551749147031086017989309766563e-02 9.46913623220784796130633509392283e-03
            1.91074763640529221575548035616521e-01 7.54170061444767725333804264664650e-01 9.46913623220784796130633509392283e-03
            2.32034277688137335893259205477079e-02 8.55220420022761995682003544061445e-03 8.27233357417523880797394220110164e-04
            2.32034277688137335893259205477079e-02 9.68244368030958590942702812753851e-01 8.27233357417523880797394220110164e-04
            8.55220420022761995682003544061445e-03 2.32034277688137335893259205477079e-02 8.27233357417523880797394220110164e-04
            8.55220420022761995682003544061445e-03 9.68244368030958590942702812753851e-01 8.27233357417523880797394220110164e-04
            9.68244368030958590942702812753851e-01 2.32034277688137335893259205477079e-02 8.27233357417523880797394220110164e-04
            9.68244368030958590942702812753851e-01 8.55220420022761995682003544061445e-03 8.27233357417523880797394220110164e-04
            1.89317782804059309142985512153246e-02 6.49303698245446425652005473239115e-01 7.50430089214290142740626521344893e-03
            1.89317782804059309142985512153246e-02 3.31764523474147643433695975545561e-01 7.50430089214290142740626521344893e-03
            6.49303698245446425652005473239115e-01 1.89317782804059309142985512153246e-02 7.50430089214290142740626521344893e-03
            6.49303698245446425652005473239115e-01 3.31764523474147643433695975545561e-01 7.50430089214290142740626521344893e-03
            3.31764523474147643433695975545561e-01 1.89317782804059309142985512153246e-02 7.50430089214290142740626521344893e-03
            3.31764523474147643433695975545561e-01 6.49303698245446425652005473239115e-01 7.50430089214290142740626521344893e-03
            1.90301297436974437005119398236275e-02 8.06961669858729524484886042046128e-02 3.97379696669624814936438994550372e-03
            1.90301297436974437005119398236275e-02 9.00273703270429548339848224713933e-01 3.97379696669624814936438994550372e-03
            8.06961669858729524484886042046128e-02 1.90301297436974437005119398236275e-02 3.97379696669624814936438994550372e-03
            8.06961669858729524484886042046128e-02 9.00273703270429548339848224713933e-01 3.97379696669624814936438994550372e-03
            9.00273703270429548339848224713933e-01 1.90301297436974437005119398236275e-02 3.97379696669624814936438994550372e-03
            9.00273703270429548339848224713933e-01 8.06961669858729524484886042046128e-02 3.97379696669624814936438994550372e-03
            1.02606190239398098462686448328895e-01 5.89148840564247877615855486510554e-01 1.59918050396850308647600513722864e-02
            1.02606190239398098462686448328895e-01 3.08244969196354023921458065160550e-01 1.59918050396850308647600513722864e-02
            5.89148840564247877615855486510554e-01 1.02606190239398098462686448328895e-01 1.59918050396850308647600513722864e-02
            5.89148840564247877615855486510554e-01 3.08244969196354023921458065160550e-01 1.59918050396850308647600513722864e-02
            3.08244969196354023921458065160550e-01 1.02606190239398098462686448328895e-01 1.59918050396850308647600513722864e-02
            3.08244969196354023921458065160550e-01 5.89148840564247877615855486510554e-01 1.59918050396850308647600513722864e-02
            5.93635001682224494601314290775917e-03 1.87441782483782071189182261150563e-01 2.69559355842440527550873952122856e-03
            5.93635001682224494601314290775917e-03 8.06621867499395683864804595941678e-01 2.69559355842440527550873952122856e-03
            1.87441782483782071189182261150563e-01 5.93635001682224494601314290775917e-03 2.69559355842440527550873952122856e-03
            1.87441782483782071189182261150563e-01 8.06621867499395683864804595941678e-01 2.69559355842440527550873952122856e-03
            8.06621867499395683864804595941678e-01 5.93635001682224494601314290775917e-03 2.69559355842440527550873952122856e-03
            8.06621867499395683864804595941678e-01 1.87441782483782071189182261150563e-01 2.69559355842440527550873952122856e-03
            ];
        
    case  17
        
        % ALG. DEG.:   17
        % PTS CARD.:   60
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.1e-16
        
        xyw=[
            4.17103444361599184908584447839530e-01 4.17103444361599184908584447839530e-01 1.36554632640510566904401557053461e-02
            4.17103444361599184908584447839530e-01 1.65793111276801630182831104320940e-01 1.36554632640510566904401557053461e-02
            1.65793111276801630182831104320940e-01 4.17103444361599184908584447839530e-01 1.36554632640510566904401557053461e-02
            1.47554916607539610140520380809903e-02 1.47554916607539610140520380809903e-02 1.38694378881882131662739432442777e-03
            1.47554916607539610140520380809903e-02 9.70489016678492077971895923838019e-01 1.38694378881882131662739432442777e-03
            9.70489016678492077971895923838019e-01 1.47554916607539610140520380809903e-02 1.38694378881882131662739432442777e-03
            4.65597871618890324363349009217927e-01 4.65597871618890324363349009217927e-01 1.25097254752486817391821105616145e-02
            4.65597871618890324363349009217927e-01 6.88042567622193512733019815641455e-02 1.25097254752486817391821105616145e-02
            6.88042567622193512733019815641455e-02 4.65597871618890324363349009217927e-01 1.25097254752486817391821105616145e-02
            1.80358116266370605007551830567536e-01 1.80358116266370605007551830567536e-01 1.31563152940089959919145456979095e-02
            1.80358116266370605007551830567536e-01 6.39283767467258789984896338864928e-01 1.31563152940089959919145456979095e-02
            6.39283767467258789984896338864928e-01 1.80358116266370605007551830567536e-01 1.31563152940089959919145456979095e-02
            6.66540634795969033632445643888786e-02 6.66540634795969033632445643888786e-02 6.22950040115272281326941339330006e-03
            6.66540634795969033632445643888786e-02 8.66691873040806193273510871222243e-01 6.22950040115272281326941339330006e-03
            8.66691873040806193273510871222243e-01 6.66540634795969033632445643888786e-02 6.22950040115272281326941339330006e-03
            2.85706502436586629034565021356684e-01 2.85706502436586629034565021356684e-01 1.88581185763976449942802560144628e-02
            2.85706502436586629034565021356684e-01 4.28586995126826741930869957286632e-01 1.88581185763976449942802560144628e-02
            4.28586995126826741930869957286632e-01 2.85706502436586629034565021356684e-01 1.88581185763976449942802560144628e-02
            1.60176423621192776103328014869476e-02 8.24790070165088096132421924266964e-01 3.98915010296479761314758505363898e-03
            1.60176423621192776103328014869476e-02 1.59192287472792681768396505503915e-01 3.98915010296479761314758505363898e-03
            8.24790070165088096132421924266964e-01 1.60176423621192776103328014869476e-02 3.98915010296479761314758505363898e-03
            8.24790070165088096132421924266964e-01 1.59192287472792681768396505503915e-01 3.98915010296479761314758505363898e-03
            1.59192287472792681768396505503915e-01 1.60176423621192776103328014869476e-02 3.98915010296479761314758505363898e-03
            1.59192287472792681768396505503915e-01 8.24790070165088096132421924266964e-01 3.98915010296479761314758505363898e-03
            3.06281591746186521163508587051183e-01 6.73493778673611709351121135114226e-02 1.12438862733455352538314286903187e-02
            3.06281591746186521163508587051183e-01 6.26369030386452307901379299437394e-01 1.12438862733455352538314286903187e-02
            6.73493778673611709351121135114226e-02 3.06281591746186521163508587051183e-01 1.12438862733455352538314286903187e-02
            6.73493778673611709351121135114226e-02 6.26369030386452307901379299437394e-01 1.12438862733455352538314286903187e-02
            6.26369030386452307901379299437394e-01 3.06281591746186521163508587051183e-01 1.12438862733455352538314286903187e-02
            6.26369030386452307901379299437394e-01 6.73493778673611709351121135114226e-02 1.12438862733455352538314286903187e-02
            1.32296727600869079566336949937977e-02 5.71294867944684092719853651942685e-01 5.19921997791976918390499662336879e-03
            1.32296727600869079566336949937977e-02 4.15475459295228999323512653063517e-01 5.19921997791976918390499662336879e-03
            5.71294867944684092719853651942685e-01 1.32296727600869079566336949937977e-02 5.19921997791976918390499662336879e-03
            5.71294867944684092719853651942685e-01 4.15475459295228999323512653063517e-01 5.19921997791976918390499662336879e-03
            4.15475459295228999323512653063517e-01 1.32296727600869079566336949937977e-02 5.19921997791976918390499662336879e-03
            4.15475459295228999323512653063517e-01 5.71294867944684092719853651942685e-01 5.19921997791976918390499662336879e-03
            7.80423405682824089524274313589558e-02 1.68722513495259462956710194703192e-01 1.02789491602272610448753908940489e-02
            7.80423405682824089524274313589558e-02 7.53235145936458128090862373937853e-01 1.02789491602272610448753908940489e-02
            1.68722513495259462956710194703192e-01 7.80423405682824089524274313589558e-02 1.02789491602272610448753908940489e-02
            1.68722513495259462956710194703192e-01 7.53235145936458128090862373937853e-01 1.02789491602272610448753908940489e-02
            7.53235145936458128090862373937853e-01 7.80423405682824089524274313589558e-02 1.02789491602272610448753908940489e-02
            7.53235145936458128090862373937853e-01 1.68722513495259462956710194703192e-01 1.02789491602272610448753908940489e-02
            1.31358708340026941741029986587819e-02 7.15072259110642427515358576783910e-01 4.34610725050059692325943316859593e-03
            1.31358708340026941741029986587819e-02 2.71791870055354878310538424557308e-01 4.34610725050059692325943316859593e-03
            7.15072259110642427515358576783910e-01 1.31358708340026941741029986587819e-02 4.34610725050059692325943316859593e-03
            7.15072259110642427515358576783910e-01 2.71791870055354878310538424557308e-01 4.34610725050059692325943316859593e-03
            2.71791870055354878310538424557308e-01 1.31358708340026941741029986587819e-02 4.34610725050059692325943316859593e-03
            2.71791870055354878310538424557308e-01 7.15072259110642427515358576783910e-01 4.34610725050059692325943316859593e-03
            1.15751759031806344957260535011301e-02 7.25054707990024360775294098857557e-02 2.29217420086793395236912651569128e-03
            1.15751759031806344957260535011301e-02 9.15919353297816929426744536613114e-01 2.29217420086793395236912651569128e-03
            7.25054707990024360775294098857557e-02 1.15751759031806344957260535011301e-02 2.29217420086793395236912651569128e-03
            7.25054707990024360775294098857557e-02 9.15919353297816929426744536613114e-01 2.29217420086793395236912651569128e-03
            9.15919353297816929426744536613114e-01 1.15751759031806344957260535011301e-02 2.29217420086793395236912651569128e-03
            9.15919353297816929426744536613114e-01 7.25054707990024360775294098857557e-02 2.29217420086793395236912651569128e-03
            1.57505477926869918992025532133994e-01 5.43275579596159796658128016133560e-01 1.30858129676684978998091324342568e-02
            1.57505477926869918992025532133994e-01 2.99218942476970339860997682990273e-01 1.30858129676684978998091324342568e-02
            5.43275579596159796658128016133560e-01 1.57505477926869918992025532133994e-01 1.30858129676684978998091324342568e-02
            5.43275579596159796658128016133560e-01 2.99218942476970339860997682990273e-01 1.30858129676684978998091324342568e-02
            2.99218942476970339860997682990273e-01 1.57505477926869918992025532133994e-01 1.30858129676684978998091324342568e-02
            2.99218942476970339860997682990273e-01 5.43275579596159796658128016133560e-01 1.30858129676684978998091324342568e-02
            ];
        
    case  18
        
        % ALG. DEG.:   18
        % PTS CARD.:   67
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 9.0e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 1.81778676507133515882763674653688e-02
            3.99955628067576229867086112790275e-01 3.99955628067576229867086112790275e-01 1.66522350166950806882315561097130e-02
            3.99955628067576229867086112790275e-01 2.00088743864847540265827774419449e-01 1.66522350166950806882315561097130e-02
            2.00088743864847540265827774419449e-01 3.99955628067576229867086112790275e-01 1.66522350166950806882315561097130e-02
            4.87580301574869534597667097841622e-01 4.87580301574869534597667097841622e-01 6.02332381699986069145680289693701e-03
            4.87580301574869534597667097841622e-01 2.48393968502609308046658043167554e-02 6.02332381699986069145680289693701e-03
            2.48393968502609308046658043167554e-02 4.87580301574869534597667097841622e-01 6.02332381699986069145680289693701e-03
            4.61809506406449243876011223619571e-01 4.61809506406449243876011223619571e-01 9.47458575338944175570077277370729e-03
            4.61809506406449243876011223619571e-01 7.63809871871015122479775527608581e-02 9.47458575338944175570077277370729e-03
            7.63809871871015122479775527608581e-02 4.61809506406449243876011223619571e-01 9.47458575338944175570077277370729e-03
            2.42264702514271956790281592475367e-01 2.42264702514271956790281592475367e-01 1.82375447044718363986959275280242e-02
            2.42264702514271956790281592475367e-01 5.15470594971456086419436815049266e-01 1.82375447044718363986959275280242e-02
            5.15470594971456086419436815049266e-01 2.42264702514271956790281592475367e-01 1.82375447044718363986959275280242e-02
            3.88302560886856218402840568160173e-02 3.88302560886856218402840568160173e-02 3.56466300985948825880145207634087e-03
            3.88302560886856218402840568160173e-02 9.22339487822628756319431886367965e-01 3.56466300985948825880145207634087e-03
            9.22339487822628756319431886367965e-01 3.88302560886856218402840568160173e-02 3.56466300985948825880145207634087e-03
            9.19477421216431944905878026474966e-02 9.19477421216431944905878026474966e-02 8.27957997600163066176381931882133e-03
            9.19477421216431944905878026474966e-02 8.16104515756713611018824394705007e-01 8.27957997600163066176381931882133e-03
            8.16104515756713611018824394705007e-01 9.19477421216431944905878026474966e-02 8.27957997600163066176381931882133e-03
            4.58049158598607952797010511858389e-02 7.70372376214675247396712620684411e-01 6.87980811747110863885179554699789e-03
            4.58049158598607952797010511858389e-02 1.83822707925463957323586328129750e-01 6.87980811747110863885179554699789e-03
            7.70372376214675247396712620684411e-01 4.58049158598607952797010511858389e-02 6.87980811747110863885179554699789e-03
            7.70372376214675247396712620684411e-01 1.83822707925463957323586328129750e-01 6.87980811747110863885179554699789e-03
            1.83822707925463957323586328129750e-01 4.58049158598607952797010511858389e-02 6.87980811747110863885179554699789e-03
            1.83822707925463957323586328129750e-01 7.70372376214675247396712620684411e-01 6.87980811747110863885179554699789e-03
            2.06349257433837918185304260987323e-01 1.22696757371927533952771227632184e-01 1.18909554500764257090050790566238e-02
            2.06349257433837918185304260987323e-01 6.70953985194234547861924511380494e-01 1.18909554500764257090050790566238e-02
            1.22696757371927533952771227632184e-01 2.06349257433837918185304260987323e-01 1.18909554500764257090050790566238e-02
            1.22696757371927533952771227632184e-01 6.70953985194234547861924511380494e-01 1.18909554500764257090050790566238e-02
            6.70953985194234547861924511380494e-01 2.06349257433837918185304260987323e-01 1.18909554500764257090050790566238e-02
            6.70953985194234547861924511380494e-01 1.22696757371927533952771227632184e-01 1.18909554500764257090050790566238e-02
            3.89761103347335824409469751117285e-03 6.00418954634256873958975120331161e-01 2.26526725112853469581986054492972e-03
            3.89761103347335824409469751117285e-03 3.95683434332269712285778950899839e-01 2.26526725112853469581986054492972e-03
            6.00418954634256873958975120331161e-01 3.89761103347335824409469751117285e-03 2.26526725112853469581986054492972e-03
            6.00418954634256873958975120331161e-01 3.95683434332269712285778950899839e-01 2.26526725112853469581986054492972e-03
            3.95683434332269712285778950899839e-01 3.89761103347335824409469751117285e-03 2.26526725112853469581986054492972e-03
            3.95683434332269712285778950899839e-01 6.00418954634256873958975120331161e-01 2.26526725112853469581986054492972e-03
            1.34620167414449820597610596450977e-02 1.08195793791033278985480592382373e-01 3.42005505980359390469369706977432e-03
            1.34620167414449820597610596450977e-02 8.78342189467521738954758347972529e-01 3.42005505980359390469369706977432e-03
            1.08195793791033278985480592382373e-01 1.34620167414449820597610596450977e-02 3.42005505980359390469369706977432e-03
            1.08195793791033278985480592382373e-01 8.78342189467521738954758347972529e-01 3.42005505980359390469369706977432e-03
            8.78342189467521738954758347972529e-01 1.34620167414449820597610596450977e-02 3.42005505980359390469369706977432e-03
            8.78342189467521738954758347972529e-01 1.08195793791033278985480592382373e-01 3.42005505980359390469369706977432e-03
            4.02602834699080647240521102503408e-02 6.39988092004714625993244681012584e-01 8.87374455101021079872491981177518e-03
            4.02602834699080647240521102503408e-02 3.19751624525377309282703208737075e-01 8.87374455101021079872491981177518e-03
            6.39988092004714625993244681012584e-01 4.02602834699080647240521102503408e-02 8.87374455101021079872491981177518e-03
            6.39988092004714625993244681012584e-01 3.19751624525377309282703208737075e-01 8.87374455101021079872491981177518e-03
            3.19751624525377309282703208737075e-01 4.02602834699080647240521102503408e-02 8.87374455101021079872491981177518e-03
            3.19751624525377309282703208737075e-01 6.39988092004714625993244681012584e-01 8.87374455101021079872491981177518e-03
            5.29833518660977054182126266823616e-03 2.35772184958191743930910888593644e-01 2.50533043728986323100960120768832e-03
            5.29833518660977054182126266823616e-03 7.58929479855198430016116617480293e-01 2.50533043728986323100960120768832e-03
            2.35772184958191743930910888593644e-01 5.29833518660977054182126266823616e-03 2.50533043728986323100960120768832e-03
            2.35772184958191743930910888593644e-01 7.58929479855198430016116617480293e-01 2.50533043728986323100960120768832e-03
            7.58929479855198430016116617480293e-01 5.29833518660977054182126266823616e-03 2.50533043728986323100960120768832e-03
            7.58929479855198430016116617480293e-01 2.35772184958191743930910888593644e-01 2.50533043728986323100960120768832e-03
            5.48360042042339568268971561337821e-04 9.72360728962795684005016028095270e-01 6.11474063480545453226622232278942e-04
            5.48360042042339568268971561337821e-04 2.70909109951619209155637690855656e-02 6.11474063480545453226622232278942e-04
            9.72360728962795684005016028095270e-01 5.48360042042339568268971561337821e-04 6.11474063480545453226622232278942e-04
            9.72360728962795684005016028095270e-01 2.70909109951619209155637690855656e-02 6.11474063480545453226622232278942e-04
            2.70909109951619209155637690855656e-02 5.48360042042339568268971561337821e-04 6.11474063480545453226622232278942e-04
            2.70909109951619209155637690855656e-02 9.72360728962795684005016028095270e-01 6.11474063480545453226622232278942e-04
            1.20587695163924646379882688052021e-01 3.33493529449880754533808158157626e-01 1.27410876559122324125361558344594e-02
            1.20587695163924646379882688052021e-01 5.45918775386194599086309153790353e-01 1.27410876559122324125361558344594e-02
            3.33493529449880754533808158157626e-01 1.20587695163924646379882688052021e-01 1.27410876559122324125361558344594e-02
            3.33493529449880754533808158157626e-01 5.45918775386194599086309153790353e-01 1.27410876559122324125361558344594e-02
            5.45918775386194599086309153790353e-01 1.20587695163924646379882688052021e-01 1.27410876559122324125361558344594e-02
            5.45918775386194599086309153790353e-01 3.33493529449880754533808158157626e-01 1.27410876559122324125361558344594e-02
            ];
        
    case  19
        
        % ALG. DEG.:   19
        % PTS CARD.:   73
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.4e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 1.72346988520061736305244437517104e-02
            5.25238903512089683189856259559747e-02 5.25238903512089683189856259559747e-02 3.55462829889906717015235493306591e-03
            5.25238903512089683189856259559747e-02 8.94952219297582063362028748088051e-01 3.55462829889906717015235493306591e-03
            8.94952219297582063362028748088051e-01 5.25238903512089683189856259559747e-02 3.55462829889906717015235493306591e-03
            4.92512675041336889236731622077059e-01 4.92512675041336889236731622077059e-01 5.16087757147214339081831013800183e-03
            4.92512675041336889236731622077059e-01 1.49746499173262215265367558458820e-02 5.16087757147214339081831013800183e-03
            1.49746499173262215265367558458820e-02 4.92512675041336889236731622077059e-01 5.16087757147214339081831013800183e-03
            1.11448873323021391268383695205557e-01 1.11448873323021391268383695205557e-01 7.61717554650915337072225597125907e-03
            1.11448873323021391268383695205557e-01 7.77102253353957217463232609588886e-01 7.61717554650915337072225597125907e-03
            7.77102253353957217463232609588886e-01 1.11448873323021391268383695205557e-01 7.61717554650915337072225597125907e-03
            4.59194201039543670184173151938012e-01 4.59194201039543670184173151938012e-01 1.14917950133708087617767645838285e-02
            4.59194201039543670184173151938012e-01 8.16115979209126596316536961239763e-02 1.14917950133708087617767645838285e-02
            8.16115979209126596316536961239763e-02 4.59194201039543670184173151938012e-01 1.14917950133708087617767645838285e-02
            4.03969722551901222473702546267305e-01 4.03969722551901222473702546267305e-01 1.57687674465774932408557162943907e-02
            4.03969722551901222473702546267305e-01 1.92060554896197555052594907465391e-01 1.57687674465774932408557162943907e-02
            1.92060554896197555052594907465391e-01 4.03969722551901222473702546267305e-01 1.57687674465774932408557162943907e-02
            1.78170104781764315760028694057837e-01 1.78170104781764315760028694057837e-01 1.23259574240954326157515552608857e-02
            1.78170104781764315760028694057837e-01 6.43659790436471368479942611884326e-01 1.23259574240954326157515552608857e-02
            6.43659790436471368479942611884326e-01 1.78170104781764315760028694057837e-01 1.23259574240954326157515552608857e-02
            1.16394611837894457195830000273418e-02 1.16394611837894457195830000273418e-02 8.82661388221424271158244145851768e-04
            1.16394611837894457195830000273418e-02 9.76721077632421108560833999945316e-01 8.82661388221424271158244145851768e-04
            9.76721077632421108560833999945316e-01 1.16394611837894457195830000273418e-02 8.82661388221424271158244145851768e-04
            2.55161632913607716588444418448489e-01 2.55161632913607716588444418448489e-01 1.58765096830015446649486676733432e-02
            2.55161632913607716588444418448489e-01 4.89676734172784566823111163103022e-01 1.58765096830015446649486676733432e-02
            4.89676734172784566823111163103022e-01 2.55161632913607716588444418448489e-01 1.58765096830015446649486676733432e-02
            1.30697676268032414448327926947968e-01 8.30156464400275351245284127799096e-01 4.84774224342752503569276001371691e-03
            1.30697676268032414448327926947968e-01 3.91458593316922343063879452529363e-02 4.84774224342752503569276001371691e-03
            8.30156464400275351245284127799096e-01 1.30697676268032414448327926947968e-01 4.84774224342752503569276001371691e-03
            8.30156464400275351245284127799096e-01 3.91458593316922343063879452529363e-02 4.84774224342752503569276001371691e-03
            3.91458593316922343063879452529363e-02 1.30697676268032414448327926947968e-01 4.84774224342752503569276001371691e-03
            3.91458593316922343063879452529363e-02 8.30156464400275351245284127799096e-01 4.84774224342752503569276001371691e-03
            3.11317629809541251972859754459932e-01 1.29312564470157820295526107656769e-01 1.31731609886953718313584715815523e-02
            3.11317629809541251972859754459932e-01 5.59369805720300927731614137883298e-01 1.31731609886953718313584715815523e-02
            1.29312564470157820295526107656769e-01 3.11317629809541251972859754459932e-01 1.31731609886953718313584715815523e-02
            1.29312564470157820295526107656769e-01 5.59369805720300927731614137883298e-01 1.31731609886953718313584715815523e-02
            5.59369805720300927731614137883298e-01 3.11317629809541251972859754459932e-01 1.31731609886953718313584715815523e-02
            5.59369805720300927731614137883298e-01 1.29312564470157820295526107656769e-01 1.31731609886953718313584715815523e-02
            2.06892589660478964930234724306501e-03 6.33313293128784149388366131461225e-01 1.64103827591791030966750231101514e-03
            2.06892589660478964930234724306501e-03 3.64617780974611060962331521295710e-01 1.64103827591791030966750231101514e-03
            6.33313293128784149388366131461225e-01 2.06892589660478964930234724306501e-03 1.64103827591791030966750231101514e-03
            6.33313293128784149388366131461225e-01 3.64617780974611060962331521295710e-01 1.64103827591791030966750231101514e-03
            3.64617780974611060962331521295710e-01 2.06892589660478964930234724306501e-03 1.64103827591791030966750231101514e-03
            3.64617780974611060962331521295710e-01 6.33313293128784149388366131461225e-01 1.64103827591791030966750231101514e-03
            7.45602946016266643347591980273137e-02 2.21434885432331196586375199331087e-01 9.05397246560622932787243399843646e-03
            7.45602946016266643347591980273137e-02 7.04004819966042250101168065157253e-01 9.05397246560622932787243399843646e-03
            2.21434885432331196586375199331087e-01 7.45602946016266643347591980273137e-02 9.05397246560622932787243399843646e-03
            2.21434885432331196586375199331087e-01 7.04004819966042250101168065157253e-01 9.05397246560622932787243399843646e-03
            7.04004819966042250101168065157253e-01 7.45602946016266643347591980273137e-02 9.05397246560622932787243399843646e-03
            7.04004819966042250101168065157253e-01 2.21434885432331196586375199331087e-01 9.05397246560622932787243399843646e-03
            5.00728825735446969602548961120192e-03 8.52566954376889230005076569796074e-01 1.46315755173510083503307654950731e-03
            5.00728825735446969602548961120192e-03 1.42425757365756355810049171850551e-01 1.46315755173510083503307654950731e-03
            8.52566954376889230005076569796074e-01 5.00728825735446969602548961120192e-03 1.46315755173510083503307654950731e-03
            8.52566954376889230005076569796074e-01 1.42425757365756355810049171850551e-01 1.46315755173510083503307654950731e-03
            1.42425757365756355810049171850551e-01 5.00728825735446969602548961120192e-03 1.46315755173510083503307654950731e-03
            1.42425757365756355810049171850551e-01 8.52566954376889230005076569796074e-01 1.46315755173510083503307654950731e-03
            4.08880111960168712847973893076414e-02 3.54028009735275150937638954928843e-01 8.05108138201205726647291527342531e-03
            4.08880111960168712847973893076414e-02 6.05083979068708033288714887021342e-01 8.05108138201205726647291527342531e-03
            3.54028009735275150937638954928843e-01 4.08880111960168712847973893076414e-02 8.05108138201205726647291527342531e-03
            3.54028009735275150937638954928843e-01 6.05083979068708033288714887021342e-01 8.05108138201205726647291527342531e-03
            6.05083979068708033288714887021342e-01 4.08880111960168712847973893076414e-02 8.05108138201205726647291527342531e-03
            6.05083979068708033288714887021342e-01 3.54028009735275150937638954928843e-01 8.05108138201205726647291527342531e-03
            2.41894578960579587079138264016365e-01 7.43181368957436361277757441712311e-01 4.22794374976824972184319406665054e-03
            2.41894578960579587079138264016365e-01 1.49240520819841071542555255291518e-02 4.22794374976824972184319406665054e-03
            7.43181368957436361277757441712311e-01 2.41894578960579587079138264016365e-01 4.22794374976824972184319406665054e-03
            7.43181368957436361277757441712311e-01 1.49240520819841071542555255291518e-02 4.22794374976824972184319406665054e-03
            1.49240520819841071542555255291518e-02 2.41894578960579587079138264016365e-01 4.22794374976824972184319406665054e-03
            1.49240520819841071542555255291518e-02 7.43181368957436361277757441712311e-01 4.22794374976824972184319406665054e-03
            6.00862753223067036500992799119558e-02 9.77602580088815509284927429689560e-03 1.66360068142969467694380902855755e-03
            6.00862753223067036500992799119558e-02 9.30137698876805085745900214533322e-01 1.66360068142969467694380902855755e-03
            9.77602580088815509284927429689560e-03 6.00862753223067036500992799119558e-02 1.66360068142969467694380902855755e-03
            9.77602580088815509284927429689560e-03 9.30137698876805085745900214533322e-01 1.66360068142969467694380902855755e-03
            9.30137698876805085745900214533322e-01 6.00862753223067036500992799119558e-02 1.66360068142969467694380902855755e-03
            9.30137698876805085745900214533322e-01 9.77602580088815509284927429689560e-03 1.66360068142969467694380902855755e-03
            ];
        
    case  20
        
        % ALG. DEG.:   20
        % PTS CARD.:   79
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 7.5e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 1.39101107014531280570146520858543e-02
            2.54579267673339104671725863227039e-01 2.54579267673339104671725863227039e-01 1.40832013075202593099799486253687e-02
            2.54579267673339104671725863227039e-01 4.90841464653321790656548273545923e-01 1.40832013075202593099799486253687e-02
            4.90841464653321790656548273545923e-01 2.54579267673339104671725863227039e-01 1.40832013075202593099799486253687e-02
            1.09761410283977789426046456355834e-02 1.09761410283977789426046456355834e-02 7.98840791066620617595639419761255e-04
            1.09761410283977789426046456355834e-02 9.78047717943204442114790708728833e-01 7.98840791066620617595639419761255e-04
            9.78047717943204442114790708728833e-01 1.09761410283977789426046456355834e-02 7.98840791066620617595639419761255e-04
            1.09383596711714603522125344170490e-01 1.09383596711714603522125344170490e-01 7.83023077607454022486255240664832e-03
            1.09383596711714603522125344170490e-01 7.81232806576570792955749311659019e-01 7.83023077607454022486255240664832e-03
            7.81232806576570792955749311659019e-01 1.09383596711714603522125344170490e-01 7.83023077607454022486255240664832e-03
            1.86294997744540946627012090175413e-01 1.86294997744540946627012090175413e-01 9.17346297425292341032321274951755e-03
            1.86294997744540946627012090175413e-01 6.27410004510918106745975819649175e-01 9.17346297425292341032321274951755e-03
            6.27410004510918106745975819649175e-01 1.86294997744540946627012090175413e-01 9.17346297425292341032321274951755e-03
            4.45551056955924784652722792088753e-01 4.45551056955924784652722792088753e-01 9.45239993323245680789668199395237e-03
            4.45551056955924784652722792088753e-01 1.08897886088150430694554415822495e-01 9.45239993323245680789668199395237e-03
            1.08897886088150430694554415822495e-01 4.45551056955924784652722792088753e-01 9.45239993323245680789668199395237e-03
            3.73108805988847103130012783367420e-02 3.73108805988847103130012783367420e-02 2.16127541066557906160316981925007e-03
            3.73108805988847103130012783367420e-02 9.25378238802230579373997443326516e-01 2.16127541066557906160316981925007e-03
            9.25378238802230579373997443326516e-01 3.73108805988847103130012783367420e-02 2.16127541066557906160316981925007e-03
            3.93425347817099868574786114550079e-01 3.93425347817099868574786114550079e-01 1.37880506290704706734429052517044e-02
            3.93425347817099868574786114550079e-01 2.13149304365800262850427770899842e-01 1.37880506290704706734429052517044e-02
            2.13149304365800262850427770899842e-01 3.93425347817099868574786114550079e-01 1.37880506290704706734429052517044e-02
            4.76245611540498992031444913664018e-01 4.76245611540498992031444913664018e-01 7.10182530340844678229395015023329e-03
            4.76245611540498992031444913664018e-01 4.75087769190020159371101726719644e-02 7.10182530340844678229395015023329e-03
            4.75087769190020159371101726719644e-02 4.76245611540498992031444913664018e-01 7.10182530340844678229395015023329e-03
            7.57078050469650598586213163798675e-03 8.33295511838236246937583473481936e-01 2.20289741855849959331625775860175e-03
            7.57078050469650598586213163798675e-03 1.59133707657067247076554394880077e-01 2.20289741855849959331625775860175e-03
            8.33295511838236246937583473481936e-01 7.57078050469650598586213163798675e-03 2.20289741855849959331625775860175e-03
            8.33295511838236246937583473481936e-01 1.59133707657067247076554394880077e-01 2.20289741855849959331625775860175e-03
            1.59133707657067247076554394880077e-01 7.57078050469650598586213163798675e-03 2.20289741855849959331625775860175e-03
            1.59133707657067247076554394880077e-01 8.33295511838236246937583473481936e-01 2.20289741855849959331625775860175e-03
            4.65603649076642978066331579611870e-02 1.98518132228788168891497889489983e-01 5.98639857895469536253063580488742e-03
            4.65603649076642978066331579611870e-02 7.54921502863547644324171415064484e-01 5.98639857895469536253063580488742e-03
            1.98518132228788168891497889489983e-01 4.65603649076642978066331579611870e-02 5.98639857895469536253063580488742e-03
            1.98518132228788168891497889489983e-01 7.54921502863547644324171415064484e-01 5.98639857895469536253063580488742e-03
            7.54921502863547644324171415064484e-01 4.65603649076642978066331579611870e-02 5.98639857895469536253063580488742e-03
            7.54921502863547644324171415064484e-01 1.98518132228788168891497889489983e-01 5.98639857895469536253063580488742e-03
            6.40905856084340586065195566334296e-02 9.31054476783942153161888199974783e-01 1.12986960212586667016976260669026e-03
            6.40905856084340586065195566334296e-02 4.85493760762378823159224339178763e-03 1.12986960212586667016976260669026e-03
            9.31054476783942153161888199974783e-01 6.40905856084340586065195566334296e-02 1.12986960212586667016976260669026e-03
            9.31054476783942153161888199974783e-01 4.85493760762378823159224339178763e-03 1.12986960212586667016976260669026e-03
            4.85493760762378823159224339178763e-03 6.40905856084340586065195566334296e-02 1.12986960212586667016976260669026e-03
            4.85493760762378823159224339178763e-03 9.31054476783942153161888199974783e-01 1.12986960212586667016976260669026e-03
            5.49874791429867948444609737634892e-02 3.33134817309587494271738705720054e-01 8.66722556721933982959438225179838e-03
            5.49874791429867948444609737634892e-02 6.11877703547425766394951551774284e-01 8.66722556721933982959438225179838e-03
            3.33134817309587494271738705720054e-01 5.49874791429867948444609737634892e-02 8.66722556721933982959438225179838e-03
            3.33134817309587494271738705720054e-01 6.11877703547425766394951551774284e-01 8.66722556721933982959438225179838e-03
            6.11877703547425766394951551774284e-01 5.49874791429867948444609737634892e-02 8.66722556721933982959438225179838e-03
            6.11877703547425766394951551774284e-01 3.33134817309587494271738705720054e-01 8.66722556721933982959438225179838e-03
            9.99522962881386756173185403895332e-02 8.61684018936486717521461287105922e-01 4.14571152761386129553811130676877e-03
            9.99522962881386756173185403895332e-02 3.83636847753745513500689412467182e-02 4.14571152761386129553811130676877e-03
            8.61684018936486717521461287105922e-01 9.99522962881386756173185403895332e-02 4.14571152761386129553811130676877e-03
            8.61684018936486717521461287105922e-01 3.83636847753745513500689412467182e-02 4.14571152761386129553811130676877e-03
            3.83636847753745513500689412467182e-02 9.99522962881386756173185403895332e-02 4.14571152761386129553811130676877e-03
            3.83636847753745513500689412467182e-02 8.61684018936486717521461287105922e-01 4.14571152761386129553811130676877e-03
            1.06227204720270029802975386701291e-01 2.15607057390094392079760154956602e-01 7.72260782209923703212073320401032e-03
            1.06227204720270029802975386701291e-01 6.78165737889635522606113227084279e-01 7.72260782209923703212073320401032e-03
            2.15607057390094392079760154956602e-01 1.06227204720270029802975386701291e-01 7.72260782209923703212073320401032e-03
            2.15607057390094392079760154956602e-01 6.78165737889635522606113227084279e-01 7.72260782209923703212073320401032e-03
            6.78165737889635522606113227084279e-01 1.06227204720270029802975386701291e-01 7.72260782209923703212073320401032e-03
            6.78165737889635522606113227084279e-01 2.15607057390094392079760154956602e-01 7.72260782209923703212073320401032e-03
            4.20023758816224113132875572773628e-01 5.70144692890973359133965914224973e-01 3.69568150025530129873518880856409e-03
            4.20023758816224113132875572773628e-01 9.83154829280252773315851300139911e-03 3.69568150025530129873518880856409e-03
            5.70144692890973359133965914224973e-01 4.20023758816224113132875572773628e-01 3.69568150025530129873518880856409e-03
            5.70144692890973359133965914224973e-01 9.83154829280252773315851300139911e-03 3.69568150025530129873518880856409e-03
            9.83154829280252773315851300139911e-03 4.20023758816224113132875572773628e-01 3.69568150025530129873518880856409e-03
            9.83154829280252773315851300139911e-03 5.70144692890973359133965914224973e-01 3.69568150025530129873518880856409e-03
            5.42331804172428100230263225967064e-01 1.39808071991799898192709861177718e-01 1.16917457318277476230328204565012e-02
            5.42331804172428100230263225967064e-01 3.17860123835772001577026912855217e-01 1.16917457318277476230328204565012e-02
            1.39808071991799898192709861177718e-01 5.42331804172428100230263225967064e-01 1.16917457318277476230328204565012e-02
            1.39808071991799898192709861177718e-01 3.17860123835772001577026912855217e-01 1.16917457318277476230328204565012e-02
            3.17860123835772001577026912855217e-01 5.42331804172428100230263225967064e-01 1.16917457318277476230328204565012e-02
            3.17860123835772001577026912855217e-01 1.39808071991799898192709861177718e-01 1.16917457318277476230328204565012e-02
            1.07372128560110913220171369175659e-02 2.80581411423665216808842615137110e-01 3.57820023845768818773760422402574e-03
            1.07372128560110913220171369175659e-02 7.08681375720323747380291479203152e-01 3.57820023845768818773760422402574e-03
            2.80581411423665216808842615137110e-01 1.07372128560110913220171369175659e-02 3.57820023845768818773760422402574e-03
            2.80581411423665216808842615137110e-01 7.08681375720323747380291479203152e-01 3.57820023845768818773760422402574e-03
            7.08681375720323747380291479203152e-01 1.07372128560110913220171369175659e-02 3.57820023845768818773760422402574e-03
            7.08681375720323747380291479203152e-01 2.80581411423665216808842615137110e-01 3.57820023845768818773760422402574e-03
            ];
        
        
    otherwise
        xyw=[];
end




pointset_stats=[
    1    1 1.92e-16    0    0
    4    6 1.39e-16    0    0
    5    7 1.98e-16    0    0
    6   12 2.36e-16    0    0
    7   15 3.75e-16    0    0
    8   16 8.05e-16    0    0
    9   19 3.26e-16    0    0
    10   25 4.86e-16    0    0
    11   28 1.49e-15    0    0
    12   33 6.52e-16    0    0
    13   37 5.83e-16    0    0
    14   42 4.37e-16    0    0
    15   49 5.62e-16    0    0
    16   55 6.11e-16    0    0
    17   60 4.09e-16    0    0
    18   67 9.00e-16    0    0
    19   73 6.38e-16    0    0
    20   79 7.46e-16    0    0
    ];



