
function [S1,S2,S3,pointset_stats]=set_xiao_gimbutas_compact(deg)

%--------------------------------------------------------------------
% CUBATURE RULE ON THE SIMPLEX [0 0; 0 1; 1 0] WITH ADE EQUAL TO deg.
% THE CUBATURE RULE IS DESCRIBED IN COMPACT FORM VIA ITS ROTATIONS.
%--------------------------------------------------------------------
% INPUT:
% deg: algebraic degree of precision (ADE) of the rule.
%--------------------------------------------------------------------
% OUTPUT:
% S1,S2,S3,w:
%    S1 is a matrix N1 x 3,
%    S2 is a matrix N2 x 2,
%    S3 is a matrix N3 x 1,
%    with N1, N2, N3 possibly zero.
%    The pointset is written in terms of rotations of type s1, s2, s3
%    applied resp. to S1, S2, S3, where:
%    s1(w,a,b): [a b c w; a c b w; b a c w; b c a w; c a b w; c b a w];
%            with c=1-a-b.
%    s2(w,a): [a a c w; a c a w; c a a w];
%            with c=1-2*a.
%    s3(w): [1/3 1/3 1/3 w];
%    S1 gives points of which one has to consider all the rotations of
%    type s1.
%    S2 gives points of which one has to consider all the rotations of
%    type s2.
%    S3 gives points of which one has to consider all the rotations of
%    type s3.
%    REMARK: the sum of the weights of the set extracted from S1,S2,S3,
%            for each rule, makes 1/2.
%    REMARK: in S1, S2, S3, the first component is that of the weight w.
%
% pointset_stats:
%    statistics of the pointset stored in a matrix. Each rows contains
%    degree, cardinality of the full pointset, moment error w.r.t. orthn.
%    Dubiner basis, number of points with negative weights, number of
%    points outside the domain.
%--------------------------------------------------------------------
% SOURCE:
%  H. Xiao, Z. Gimbutas.
%  "A numerical algorithm for the construction of efficient quadrature
%  rules in two and higher dimensions".
%  Computers and Mathematics with Applications 59 (2010) 663-676.
%--------------------------------------------------------------------
%  REMARK.
%  In the original paper, Xiao and Gimbutas proposed three sets with
%  different symmetries, but they do not suggest where to find them in
%  terms of files.
%  We downloaded this file from the url
%          https://cims.nyu.edu/cmcl/quadratures/quadratures.html
%  modifying it to be used directly as cubature rule.
%  It is just one of the three, i.e. the pointset with fully symmetric
%  rules.
%--------------------------------------------------------------------
% | DEG | CARD | W- | PT.O|   M.E.   |Q.|
%--------------------------------------------------------------------
% |   1 |   1 |   0 |   0 | 1.92e-16 |PI|
% |   2 |   3 |   0 |   0 | 4.20e-16 |PI|
% |   3 |   6 |   0 |   0 | 1.25e-15 |PI|
% |   4 |   6 |   0 |   0 | 1.25e-15 |PI|
% |   5 |   7 |   0 |   0 | 7.08e-16 |PI|
% |   6 |  12 |   0 |   0 | 4.30e-16 |PI|
% |   7 |  15 |   0 |   0 | 7.15e-16 |PI|
% |   8 |  16 |   0 |   0 | 9.99e-16 |PI|
% |   9 |  19 |   0 |   0 | 1.12e-15 |PI|
% |  10 |  25 |   0 |   0 | 6.66e-16 |PI|
% |  11 |  28 |   0 |   0 | 5.83e-16 |PI|
% |  12 |  33 |   0 |   0 | 8.47e-16 |PI|
% |  13 |  37 |   0 |   0 | 1.06e-15 |PI|
% |  14 |  42 |   0 |   0 | 1.14e-15 |PI|
% |  15 |  49 |   0 |   0 | 8.50e-16 |PI|
% |  16 |  55 |   0 |   0 | 2.03e-15 |PI|
% |  17 |  60 |   0 |   0 | 9.49e-16 |PI|
% |  18 |  67 |   0 |   0 | 1.50e-15 |PI|
% |  19 |  73 |   0 |   0 | 1.07e-15 |PI|
% |  20 |  79 |   0 |   0 | 1.25e-15 |PI|
% |  21 |  87 |   0 |   0 | 1.42e-15 |PI|
% |  22 |  96 |   0 |   0 | 9.99e-16 |PI|
% |  23 | 103 |   0 |   0 | 2.05e-15 |PI|
% |  24 | 112 |   0 |   0 | 9.85e-16 |PI|
% |  25 | 120 |   0 |   0 | 1.80e-15 |PI|
% |  26 | 130 |   0 |   0 | 1.96e-15 |PI|
% |  27 | 141 |   0 |   0 | 1.11e-15 |PI|
% |  28 | 150 |   0 |   0 | 1.39e-15 |PI|
% |  29 | 159 |   0 |   0 | 1.34e-15 |PI|
% |  30 | 171 |   0 |   0 | 1.64e-15 |PI|
% |  31 | 181 |   0 |   0 | 1.70e-15 |PI|
% |  32 | 193 |   0 |   0 | 1.71e-15 |PI|
% |  33 | 204 |   0 |   0 | 1.45e-15 |PI|
% |  34 | 214 |   0 |   0 | 1.87e-15 |PI|
% |  35 | 228 |   0 |   0 | 1.68e-15 |PI|
% |  36 | 243 |   0 |   0 | 1.18e-15 |PI|
% |  37 | 252 |   0 |   0 | 1.03e-15 |PI|
% |  38 | 267 |   0 |   0 | 1.85e-15 |PI|
% |  39 | 282 |   0 |   0 | 1.58e-15 |PI|
% |  40 | 295 |   0 |   0 | 1.42e-15 |PI|
% |  41 | 309 |   0 |   0 | 1.45e-15 |PI|
% |  42 | 324 |   0 |   0 | 1.44e-15 |PI|
% |  43 | 339 |   0 |   0 | 1.45e-15 |PI|
% |  44 | 354 |   0 |   0 | 1.38e-15 |PI|
% |  45 | 370 |   0 |   0 | 1.52e-15 |PI|
% |  46 | 385 |   0 |   0 | 2.44e-15 |PI|
% |  47 | 399 |   0 |   0 | 1.56e-15 |PI|
% |  48 | 423 |   0 |   0 | 1.49e-15 |PI|
% |  49 | 435 |   0 |   0 | 1.23e-15 |PI|
% |  50 | 453 |   0 |   0 | 1.39e-15 |PI|
%--------------------------------------------------------------------
% LEGEND:
%--------------------------------------------------------------------
% DEG  : algebraic degree of precision
% CARD : cardinality of the pointset
% W-   : negative weights
% PT.O : points outside the simplex [0 0; 1 0; 0 1].
% M.E. : maximum moment error in inf norm.
% Q.: quality of the rule.
%        PI: pos. weights, points in simplex
%        NI: some neg. weights, points in simplex
%        P0: pos. weights, some points out of the simplex
%        N0: some neg. weights, some points out of the simplex
%--------------------------------------------------------------------


switch deg
    
    case   1
        
        % ALG. DEG.:    1
        % PTS CARD.:    1
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.9e-16
        
        S1=[];
        
        S2=[];
        
        S3=[
            5.00000000000000000000000000000000e-01
            ];
        
        
    case   2
        
        % ALG. DEG.:    2
        % PTS CARD.:    3
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.2e-16
        
        S1=[];
        
        S2=[
            1.66666666666666657414808128123695e-01 1.66666666666666601903656896865868e-01
            ];
        
        S3=[];
        
        
    case   3
        
        % ALG. DEG.:    3
        % PTS CARD.:    6
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.2e-15
        
        S1=[];
        
        S2=[
            1.11690794839005749783389376261766e-01 4.45948490915964834702123198439949e-01
            5.49758718276609423258882713980711e-02 9.15762135097706597708366871302132e-02
            ];
        
        S3=[];
        
        
    case   4
        
        % ALG. DEG.:    4
        % PTS CARD.:    6
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.2e-15
        
        S1=[];
        
        S2=[
            1.11690794839005749783389376261766e-01 4.45948490915964834702123198439949e-01
            5.49758718276609423258882713980711e-02 9.15762135097706597708366871302132e-02
            ];
        
        S3=[];
        
        
    case   5
        
        % ALG. DEG.:    5
        % PTS CARD.:    7
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 7.1e-16
        
        S1=[];
        
        S2=[
            6.29695902724135836425745083033689e-02 1.01286507323456259621607955523359e-01
            6.61970763942530959766941123234574e-02 4.70142064105115053962435922585428e-01
            ];
        
        S3=[
            1.12500000000000016653345369377348e-01
            ];
        
        
    case   6
        
        % ALG. DEG.:    6
        % PTS CARD.:   12
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.3e-16
        
        S1=[
            2.03172798968303258693168800164131e-02 1.41619015923968138981337006043759e-01 1.93717243612408114572520645424447e-02
            ];
        
        S2=[
            8.56665620764904961470520561306330e-02 2.19429982549782909284630250112968e-01
            4.03655447965154817735466963313229e-02 4.80137964112215043233078404227854e-01
            ];
        
        S3=[];
        
        
    case   7
        
        % ALG. DEG.:    7
        % PTS CARD.:   15
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 7.1e-16
        
        S1=[
            2.78772702703455436956669899473127e-02 2.59339011865785673549567036388908e-01 4.69712061300855548973132158607768e-02
            ];
        
        S2=[
            2.65904166483802266440772399391790e-02 4.73195653689251038986895991911297e-01
            2.04590851970284336169836336694061e-02 5.77976400545063204172713255957206e-02
            6.38626242805669236402010824349418e-02 2.41663606397247349155676943155413e-01
            ];
        
        S3=[];
        
        
    case   8
        
        % ALG. DEG.:    8
        % PTS CARD.:   16
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.0e-15
        
        S1=[
            1.36151570872174963733325370185412e-02 2.63112829634638056841566822185996e-01 8.39477740995767801601701307845360e-03
            ];
        
        S2=[
            5.16086852673591223172877562319627e-02 1.70569307751760101954374704291695e-01
            4.75458171336423096597911808203207e-02 4.59292588292723125142913431773195e-01
            1.62292488115990396480015078850556e-02 5.05472283170308595012798491552530e-02
            ];
        
        S3=[
            7.21578038388935860680817313550506e-02
            ];
        
        
    case   9
        
        % ALG. DEG.:    9
        % PTS CARD.:   19
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.1e-15
        
        S1=[
            2.16417696886446880855103813701135e-02 2.21962989160765705731748198559217e-01 3.68384120547363136388696602807613e-02
            ];
        
        S2=[
            1.56673501135695392161295558253187e-02 4.89682519198737620236272505280795e-01
            3.98238694636051382413910459945328e-02 1.88203535619032635839431577551295e-01
            3.89137705023871391385448248456669e-02 4.37089591492936579975037147960393e-01
            1.27888378293490173609470161864010e-02 4.47295133944525663549640626115433e-02
            ];
        
        S3=[
            4.85678981413994251270516144813882e-02
            ];
        
        
    case  10
        
        % ALG. DEG.:   10
        % PTS CARD.:   25
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.7e-16
        
        S1=[
            1.44811407316281712254202318490570e-02 1.33734755100869129318397199313040e-01 3.47236204823274940967081647613668e-02
            1.93695245430094524496933416912725e-02 3.26693136281336893311788571736543e-01 3.75827273411917031631368502075929e-02
            ];
        
        S2=[
            4.89629524920915173979985013374971e-03 4.95173459801170445793161434266949e-01
            3.19267961505932760385428892391246e-03 1.91394152428411012945019820108428e-02
            3.93168848731886427794179894590343e-02 1.84485012685246418939755130850244e-01
            3.76236639842719988724439872385119e-02 4.28234820943718841323999413361889e-01
            ];
        
        S3=[
            4.18074371869869634843475125762779e-02
            ];
        
        
    case  11
        
        % ALG. DEG.:   11
        % PTS CARD.:   28
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.8e-16
        
        S1=[
            7.27881166890462401319883412043055e-03 1.59303619837693477556683774309931e-01 1.43666625695556288838661274098740e-02
            2.03214243279432361932723694053493e-02 3.10631216313463132117078657756792e-01 4.76640669721508022926848013867129e-02
            ];
        
        S2=[
            6.12464847535398176914256396230485e-03 3.08468956355878905217782204317700e-02
            6.23274593694069130311863347060353e-03 4.98780165178460743735655569253140e-01
            2.00646211906541568903428185421944e-02 1.13207827286693873958256517653354e-01
            3.15474360799493439011520479198225e-02 4.36655016393176054201319402636727e-01
            3.39225538718475744248692649307486e-02 2.14483458619269262701578782071010e-01
            ];
        
        S3=[
            4.07225673546756442022775956957048e-02
            ];
        
        
    case  12
        
        % ALG. DEG.:   12
        % PTS CARD.:   33
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 8.5e-16
        
        S1=[
            2.16136818297070938676984752646604e-02 2.55454228638517299998511589365080e-01 1.16296019677926645852217291121633e-01
            7.54183878825571713672148987939181e-03 1.27279717233589356739642539650958e-01 2.13824902561706338410818517559164e-02
            1.08917925193037726933065201251338e-02 2.91655679738340944950891753251199e-01 2.30341563552671768633217652677558e-02
            ];
        
        S2=[
            3.12706065979513683772417209638661e-02 2.71462507014926024417889038886642e-01
            1.42430260344387667192922464209914e-02 1.09257827659354198046592898663221e-01
            2.49591674640304642118771027980983e-02 4.40111648658593090921442581020528e-01
            1.21334190407260122945842795161298e-02 4.88203750945541525840809526926023e-01
            3.96582125498681770103814869798953e-03 2.46463634363354618106445315106612e-02
            ];
        
        S3=[];
        
        
    case  13
        
        % ALG. DEG.:   13
        % PTS CARD.:   37
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.1e-15
        
        S1=[
            9.06274932310043891681150540762246e-03 2.92078688576636380513917856660555e-01 1.89880043837590481436983225194126e-02
            1.86059802286307612861904203782615e-02 2.66745253310351149167445328203030e-01 9.77360310660165543250599284874625e-02
            7.69653634189108610402785259907432e-03 1.26799775783837304077295016213611e-01 2.19663442065292507932028343020647e-02
            ];
        
        S2=[
            4.97073818053629309932084368028882e-03 4.96135894741046079570168103600736e-01
            1.63906208018614815702385101303662e-02 4.69608689653491895299453062762041e-01
            2.30312047963891174118256088831913e-02 2.31110284949082200878223147810786e-01
            2.34735477710775880533233817004657e-02 4.14477570279054519097883257927606e-01
            1.54515489878798931849823716788705e-02 1.13559912572133075991942519067379e-01
            4.01469989762920972248272377669309e-03 2.48959314912162443800269784333068e-02
            ];
        
        S3=[
            2.58113233321454020596785738916878e-02
            ];
        
        
    case  14
        
        % ALG. DEG.:   14
        % PTS CARD.:   42
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.1e-15
        
        S1=[
            7.21815405676692369019153971976266e-03 2.98372882136257677743174099305179e-01 1.46469500556544764136024383560652e-02
            1.92857553935303488445995867550664e-02 3.36861459796345019679364440889913e-01 9.29162493569718889174779974382545e-02
            1.23328766062818454690708236398677e-02 1.72266687821355540810586148836592e-01 5.71247574036480090797773812028026e-02
            2.50511441925033726332849859375074e-03 1.18974497696956824088765358737874e-01 1.26833093287207670021854433173303e-03
            ];
        
        S2=[
            1.63941767720626845050713171758616e-02 4.17644719340453884903752168611391e-01
            7.21684983488833729087241763977545e-03 6.17998830908724830490719170938974e-02
            2.58870522536458064211117857666977e-02 2.73477528308838591097895687198616e-01
            2.10812943684965219126592472775883e-02 1.77205532412543387277281681235763e-01
            2.46170180120004224166851969357594e-03 1.93909612487008810310928907938433e-02
            1.09417906847144516535585978544987e-02 4.88963910362178622026618768359185e-01
            ];
        
        S3=[];
        
        
    case  15
        
        % ALG. DEG.:   15
        % PTS CARD.:   49
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 8.5e-16
        
        S1=[
            1.21150043915628034124631895451785e-02 1.82321783407191295722782342636492e-01 8.45942214821918392342325887511834e-02
            5.61425214943902899100702796886253e-03 1.50200384065238723430013578763464e-01 1.60270897863454761522739033807738e-02
            1.55376102352554733981859769187395e-02 3.23111315163712664588757661476848e-01 9.76504424302423873482936755863193e-02
            8.21838104641394590066383329940436e-03 3.07947681483672908164805903652450e-01 1.84542519046331753018730381654677e-02
            1.23763300727895773738207463310346e-03 3.80352293011092870700373680392659e-02 1.11353527401374216738993983000228e-03
            ];
        
        S2=[
            3.69875203352304938314576432389913e-03 1.29978229933077754632009259694314e-01
            1.07970439682192245051117751586389e-02 4.60076949297059667465248367079766e-01
            7.91613817501089778172485011964454e-03 4.91685816630297201079713431681739e-01
            2.31436430525990381734668943636279e-02 2.21532340795141918787791723843839e-01
            2.31680206956036138787879963274463e-02 3.96933737409060571899033220688580e-01
            7.54223712379853330339818384686623e-03 5.63419176960999884551384297992627e-02
            ];
        
        S3=[
            1.48652098740356588407740545676461e-02
            ];
        
        
    case  16
        
        % ALG. DEG.:   16
        % PTS CARD.:   55
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 2.0e-15
        
        S1=[
            4.09110527661106938152046907930526e-03 4.13769485827085170548400583356852e-01 9.66495440366025790335235967631888e-03
            6.99180356232678365535626596738439e-03 3.04179448229479731935498421080410e-01 3.03059433551863786904245756659293e-02
            2.87593498524857989900871402255689e-03 8.96090890227058400885340461172746e-02 1.08129727761037564698343160785043e-02
            1.58230308409916253786775541811949e-02 2.96615372400382937279061934532365e-01 1.06653160536148483572560508036986e-01
            8.82654052355164378540486325164238e-03 1.69763355150289729778023684048094e-01 5.13543153440131280862424034694413e-02
            2.30734531986456726723733190453913e-03 2.14048779925847282301631935297337e-01 3.69694270735561368937394099987159e-03
            ];
        
        S2=[
            6.21271279778050529440935889624598e-03 6.66744722402381612136323951744998e-02
            2.05920205348962756519703987123648e-02 2.41321680701378271649559792422224e-01
            2.04926098934076865443199011451725e-02 4.12798095955223653952259610377951e-01
            1.43917483513744566547831738034802e-02 1.50063736587034979308441506873351e-01
            1.35468347338552272396405840027001e-02 4.69548030996684961912990274868207e-01
            1.89456761913211124345735036911265e-03 1.70416294057182740595557390861359e-02
            ];
        
        S3=[
            2.31139551570956752746255347119586e-02
            ];
        
        
    case  17
        
        % ALG. DEG.:   17
        % PTS CARD.:   60
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 9.5e-16
        
        S1=[
            2.29217420086793351868825752148950e-03 7.25054707990023805663781786279287e-02 1.15751759031806882721538087821500e-02
            5.19921997791976831654325863496524e-03 4.15475459295229054834663884321344e-01 1.32296727600869565288910223443963e-02
            4.34610725050059518853595719178884e-03 2.71791870055354767288235962041654e-01 1.31358708340027583588716098006444e-02
            1.30858129676684926956387045038355e-02 2.99218942476970284349846451732446e-01 1.57505477926869974503176763391821e-01
            1.12438862733455335191079527135116e-02 3.06281591746186521163508587051183e-01 6.73493778673612542018389603981632e-02
            1.02789491602272575754284389404347e-02 1.68722513495259435201134579074278e-01 7.80423405682824783413664704312396e-02
            3.98915010296479674578584706523543e-03 1.59192287472792681768396505503915e-01 1.60176423621193435298248886056172e-02
            ];
        
        S2=[
            1.36554632640510532209932037517319e-02 4.17103444361599184908584447839530e-01
            1.31563152940089925224675937442953e-02 1.80358116266370549496400599309709e-01
            1.88581185763976415248333040608486e-02 2.85706502436586573523413790098857e-01
            6.22950040115272107854593741649296e-03 6.66540634795968062187299096876814e-02
            1.38694378881882109978695982732688e-03 1.47554916607538222361739599364228e-02
            1.25097254752486782697351586080003e-02 4.65597871618890268852197777960100e-01
            ];
        
        S3=[];
        
        
    case  18
        
        % ALG. DEG.:   18
        % PTS CARD.:   67
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.5e-15
        
        S1=[
            7.66412909727656223302094673499596e-03 3.85044034413163649421818490736769e-01 9.04270403543406542379301527034841e-02
            2.10875838737221944427235875707538e-03 4.72761418326517468546654754391056e-02 1.24989324834954822901478266317099e-02
            8.18295420699327417934920703146418e-03 3.02061957712870754466649714231608e-01 5.40117353390242999067183404804382e-02
            3.86491764000310980206598365782611e-03 2.56506159774241515592052564898040e-01 1.05050188192419655802689604229272e-02
            8.45582695874002969460558887249135e-03 1.78479125565887630777694994321791e-01 6.61224580284034413324434353853576e-02
            1.37964432442897241143375808292149e-02 2.68573306396013788965859703239403e-01 1.49066910125773915973823591230030e-01
            4.79306223718074710798964233049446e-03 4.11065668674618356348560155311134e-01 1.16918246746671606467460335920805e-02
            3.82085248635981398340755177400752e-03 1.32778830271389286510341776192945e-01 1.43315247789419925039666736665822e-02
            ];
        
        S2=[
            6.55351374586937186761392837297535e-03 4.74918211324045713705999105513911e-01
            1.01591694227291893387610244303687e-02 1.51638506972604725220321597589646e-01
            1.67359970299239303737071082878174e-02 4.11067101875919438391804305865662e-01
            1.55581983010030493880249125027149e-02 2.65614609905374166043401373826782e-01
            2.66002808473890040366627296464230e-04 3.75894434106820423541206466211406e-03
            6.89514330238346316581665718103977e-03 7.24387055673327562299590454131248e-02
            ];
        
        S3=[
            1.53742606195579126215911003328074e-02
            ];
        
        
    case  19
        
        % ALG. DEG.:   19
        % PTS CARD.:   73
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.1e-15
        
        S1=[
            1.46284624394003534043895431437932e-03 1.42422282571126906924874333526532e-01 5.00514235235043503796648067805108e-03
            1.66369442029695203955075299973032e-03 6.00838999627023515048840351937542e-02 9.77706143867685707748993451104980e-03
            4.84775954081209919238615313474838e-03 1.30700669960534532343388036679244e-01 3.91424494346088655549742441053240e-02
            1.31731323537226784753384478676708e-02 3.11318383223986860919296759675490e-01 1.29312809767979042252505905707949e-01
            9.05403729521524897050355207284156e-03 2.21433941889113417023082774903742e-01 7.45611893043551660076317944003677e-02
            8.05110473046971239496905070609500e-03 3.54025926999711915144075646821875e-01 4.08883144649781413160205545409553e-02
            4.22796241954673853585822129730332e-03 2.41894104006892624036950678600988e-01 1.49236389074384866865186083373374e-02
            1.64106875741986844505215792366926e-03 3.64620414338710008284039076897898e-01 2.06910384910238874869858882732387e-03
            ];
        
        S2=[
            3.55469681139747263262651522097713e-03 5.25262798541034220400902654546371e-02
            7.61747825850241609690893795914235e-03 1.11448055716998548070328922676708e-01
            8.82596209154269781445301124023217e-04 1.16390273279224441133061773712143e-02
            1.58764272937649865424791784107583e-02 2.55162133153124748297102541982895e-01
            1.57686793226198031236950924949269e-02 4.03969717966386032692582830350148e-01
            1.23259905267924119859346276939505e-02 1.78171006079627408658083709269704e-01
            1.14917854885616226023881125684056e-02 4.59194388956827626024903565848945e-01
            5.16094109120942938495613105942539e-03 4.92512449865874224474993070543860e-01
            ];
        
        S3=[
            1.72345804254526308030293080264528e-02
            ];
        
        
    case  20
        
        % ALG. DEG.:   20
        % PTS CARD.:   79
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.2e-15
        
        S1=[
            1.12986960212586536912715562408493e-03 6.40905856084340308509439410045161e-02 4.85493760762382986495566683515790e-03
            7.72260782209923009322682929678194e-03 2.15607057390094364324184539327689e-01 1.06227204720270099191914425773575e-01
            2.20289741855849742491191278759288e-03 1.59133707657067219320978779251163e-01 7.57078050469658144633333662909536e-03
            1.16917457318277354799684886188516e-02 3.17860123835771946065875681597390e-01 1.39808071991799981459436708064459e-01
            5.98639857895469015836020787446614e-03 1.98518132228788168891497889489983e-01 4.65603649076643533177843892190140e-02
            4.14571152761385782609115935315458e-03 9.99522962881386201061673091317061e-02 3.83636847753746762501592115768290e-02
            3.69568150025529782928823685494990e-03 4.20023758816224057621724341515801e-01 9.83154829280259191792712414326161e-03
            8.66722556721933289070047834457000e-03 3.33134817309587438760587474462227e-01 5.49874791429868711722939167430013e-02
            3.57820023845768515197152126461333e-03 2.80581411423665216808842615137110e-01 1.07372128560111156081458005928653e-02
            ];
        
        S2=[
            9.17346297425291300198235688867499e-03 1.86294997744540891115860858917586e-01
            2.16127541066557732687969384244298e-03 3.73108805988845854129110080066312e-02
            7.10182530340844071076178423140846e-03 4.76245611540498992031444913664018e-01
            9.45239993323244813427930210991690e-03 4.45551056955924784652722792088753e-01
            1.40832013075202471669156167877190e-02 2.54579267673339049160574631969212e-01
            1.37880506290704567956550974372476e-02 3.93425347817099813063634883292252e-01
            7.98840791066619858654118679908152e-04 1.09761410283976314911091876069804e-02
            7.83023077607453328596864849941994e-03 1.09383596711714464744247266025923e-01
            ];
        
        S3=[
            1.39101107014531141792268442713976e-02
            ];
        
        
    case  21
        
        % ALG. DEG.:   21
        % PTS CARD.:   87
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.4e-15
        
        S1=[
            8.74770807788156565254489294147788e-03 2.89189496078594676653494843776571e-01 2.05295559335161609260111958974448e-01
            2.10306014407486523143386847323200e-03 2.37873382597993981990569523077284e-01 6.93180903146811862314269347962181e-03
            9.22374242396642105012993795298826e-03 3.18865310794828216156560074523441e-01 1.23779400405492817638375413480389e-01
            5.23495209266242305518046507017971e-03 2.31873625370400959777583693721681e-01 3.89913626232203450183710913279356e-02
            2.24040656095073845849308113997722e-03 1.33167122941370286159923352897749e-01 9.53624752971060293815774855374912e-03
            7.25015295948551290183825557278396e-03 3.46807979809911015589563021421782e-01 5.30521917012168400806793044921505e-02
            7.95201835271398815851995323100709e-03 2.16599623189982493087768489203881e-01 1.00458020074114501296591583923146e-01
            4.90598591127520706339781852989290e-03 1.28829807962051540171088959141343e-01 4.94510655685405711712299137161608e-02
            3.41994242896715393287276896217008e-03 3.60953408018922172750819754583063e-01 1.02546358729245206536928591845026e-02
            1.63271429202204300216993981820224e-03 5.57195650723719473895734211055242e-02 1.03019036434239090460618371025703e-02
            ];
        
        S2=[
            1.07255609645661737633348664644473e-02 2.98936235314982479138734561274759e-01
            2.21891484853294030862858043917640e-03 4.97007875468685578201899488703930e-01
            1.15003523266419358656031945997711e-02 4.03617586546385076484710907607223e-01
            6.82801622611510158900038902629603e-03 1.18988577622719290949326875761471e-01
            9.72762093037535940931448408264259e-03 1.90288718091278424848766803734179e-01
            6.10720508169219390703919003726696e-03 4.81597868653216576451114860901725e-01
            9.80723761391201313830645602820368e-03 4.49812791775362375013003202184336e-01
            3.57604255064182747006484852647645e-03 5.36275755461448552785519439112250e-02
            7.54349636189344881394280850628320e-04 1.07424564328282810821146853186292e-02
            ];
        
        S3=[];
        
        
    case  22
        
        % ALG. DEG.:   22
        % PTS CARD.:   96
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.0e-15
        
        S1=[
            1.29771923711563888503839336863166e-03 6.98421694674436210892309873088379e-02 7.87628222158237752126286324028115e-03
            3.75878890889418853030190525998933e-03 9.03988311664077337503897524584318e-02 4.47522843483358911953473580069840e-02
            5.59865673598138567645943197703673e-03 4.11341764020558720016396136998083e-01 3.82752347008638374448175056841137e-02
            8.85954674475510940701017403853257e-03 3.32106105007446361643275167807587e-01 1.02747075986931432622917270691687e-01
            2.45213019877848267119579972472820e-03 3.62576280432467257242024061270058e-01 7.40024123471075441493827185013288e-03
            1.08532097777544796896043166611889e-02 2.90066824116668786626149767471361e-01 1.91081297966720159431375236636086e-01
            5.83111143367150142058363826436107e-03 2.87931802824171800825325817640987e-01 4.39916453934558610017191426777572e-02
            7.85508131128515897645403498472660e-03 2.16786933364941147850402103358647e-01 1.08689941862672029282421704010630e-01
            2.05334353578777786819564710185659e-03 1.45873719873525181656148674846918e-01 9.14471137496405787892239658276594e-03
            5.28179248387344955212308050818137e-03 1.76297434824500054428852990895393e-01 4.82549241146414051306656745055079e-02
            2.52703844879230074549347762058460e-03 2.43990646039493053187641180556966e-01 9.16390924818523237849188234349640e-03
            3.20214265585712961736247761379559e-04 1.79343210529389861240812109599574e-02 1.79846498894837506649091452004541e-03
            ];
        
        S2=[
            6.74654194180533163527346829368980e-03 3.85184524627302127264272257889388e-01
            6.93069976211709709579267268964031e-03 4.57769411367672074408829985259217e-01
            1.05378819787260921692073623034958e-02 2.94558259029950031404609944729600e-01
            8.01064956257444699128100751295278e-03 1.88510523630283838292243103751389e-01
            9.42654627692064359112400495632755e-03 4.21981888793534931636486362549476e-01
            2.64466983299220959741093750494656e-03 4.96161178409708636394270797609352e-01
            1.78455458292818840322635498552017e-03 2.91084706708073584491280882957653e-02
            7.20785656405230259341498921799030e-03 1.15431538219204890549640651897789e-01
            ];
        
        S3=[];
        
        
    case  23
        
        % ALG. DEG.:   23
        % PTS CARD.:  103
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 2.1e-15
        
        S1=[
            1.26408302769113111904686341802062e-03 1.59503798924757217436720679870632e-01 2.38702536543536174284962925185027e-02
            1.11250986486225670463179326930003e-03 1.14101360322364536736827744789480e-01 5.18982176084453739051127740822267e-03
            2.66401521559739111466580396836434e-03 9.55398781717348988262727971232380e-02 3.27410291887064164306764268985717e-02
            1.14055183812791674329012447941523e-03 3.11162268051701940141384739035857e-01 2.44759985596638019594850632643102e-03
            2.05737517220804487505514401846085e-03 2.05617232058052074039622425516427e-01 8.72528958530853666897275644487308e-03
            9.76295663945362626919000081926470e-04 4.72616294497252981132717764012341e-02 7.16253991024448470603669392176016e-03
            7.49055669659957987921439226397524e-03 3.58509593569625084619190147350309e-01 6.85269541872130100168547528483032e-02
            8.06062081850857266307475157418594e-03 2.40482772035012670741593865386676e-01 1.01728329327284261474062532215612e-01
            5.23512824656503002879182062656582e-03 1.72932303129223974291761578570004e-01 5.83515752375154614361996152638312e-02
            1.04221979294844002694730988878291e-02 3.16304307653838101543897209921852e-01 1.54830155405516284305988961023104e-01
            3.54888941726091023537437152413077e-03 3.97758576803007635191988811129704e-01 1.47589697299451757389610051518503e-02
            5.08778732835351611696328433254166e-03 2.78794169814102266169442145837820e-01 3.29937081925328057341317844475270e-02
            ];
        
        S2=[
            1.95787012951646657160642561734676e-03 3.90072687570319903693771834696236e-02
            5.69894463390037669253684882164634e-03 4.80328877337308501882517930425820e-01
            4.47995851275676936625869117847287e-03 8.68410482076330536571617813024204e-02
            1.18373042315640039645607828333596e-02 3.94323506011541535798414770397358e-01
            1.19039314437498749271737707999819e-02 2.66251317877247206666879719705321e-01
            7.27972469637086939259207341024194e-03 1.37129387311647477076803625095636e-01
            1.20377230209070431969642367420192e-03 4.98959431209586301569913757703034e-01
            9.47597533466943732494147667466677e-03 4.44692442127727516343327351933112e-01
            9.96763894005250720986577306348408e-03 1.98749806396536110586481527207070e-01
            5.32680616414657275535515790210184e-04 9.01644020559819528437550673061196e-03
            ];
        
        S3=[
            1.26265301615180979000196970218894e-02
            ];
        
        
    case  24
        
        % ALG. DEG.:   24
        % PTS CARD.:  112
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 9.9e-16
        
        S1=[
            7.07252290324242305352697712805821e-03 2.41479760073594007518948956203531e-01 1.70367282462443730972623257002851e-01
            7.63722130066231321560143996407533e-03 3.28975808924226342799812528028269e-01 1.69759795860736056871687083003053e-01
            2.68313572708388410614777086493632e-03 9.31674097798811384762984744156711e-02 3.83182258210193957270739417708683e-02
            7.51592717487067126652666715358464e-03 3.94520279800194328956308709166478e-01 9.26564815207575470656919947032293e-02
            3.60206708739871416316558061510023e-03 1.62677416394477414307573326368583e-01 4.11887142484753873694636183699913e-02
            4.45243846408178295392943368824490e-03 2.53589014218879471584955354046542e-01 3.95709049701580550806312430722755e-02
            4.97362593784120982087770812540839e-03 3.62252241317791268127024295608862e-01 3.85927001748961398530646249582787e-02
            7.17617578907872762095099261614450e-03 2.81622577706160837074378378019901e-01 9.45349617365990324557856183673721e-02
            2.12107463340188446415668366284990e-03 3.83272664992659217730874843255151e-01 7.38799463229424056148753052752909e-03
            2.04063753855822545280052082716793e-03 2.73750352516260508917156357711065e-01 7.54600316231281847967338194393960e-03
            1.29460619119899241445670323713557e-03 9.41213427973660310410508600398316e-02 7.23455845778213976066117751884121e-03
            5.92178107127155472361179633367101e-03 1.80396151886765720995242645585677e-01 9.55662695273652579297518627754471e-02
            1.85361338212315434822996529362626e-03 1.74737346282805683372885141579900e-01 7.98792188084796773994700203047614e-03
            8.98473792723288272425719025449098e-04 3.72914720512912134520178142338409e-02 8.07491087020877768010151953603781e-03
            ];
        
        S2=[
            6.55526635094262014741284261276633e-03 4.18890974910602731373643337065005e-01
            5.18950802820009656629585137466165e-03 1.62360633716926272551717147507588e-01
            1.91684986546459168628431246617083e-03 4.09856290011169493991971535251650e-02
            3.08627252748321578238593909659926e-04 6.73127088788825697629469857474760e-03
            2.17162336108534881687348594425657e-03 4.96255277675735129161438408118556e-01
            1.02600043357549217809543407042838e-02 2.64231315438272496010085887974128e-01
            5.17624738542630125764931747767150e-03 4.80612561792503223401240575185511e-01
            5.01369653369445385932001713058526e-03 9.63284955992150526693507117670379e-02
            9.49729325867632902247184034649763e-03 3.75352926702086253651913239082205e-01
            ];
        
        S3=[
            6.27284492280015971293893883853343e-03
            ];
        
        
    case  25
        
        % ALG. DEG.:   25
        % PTS CARD.:  120
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.8e-15
        
        S1=[
            8.37408915967352627911235440905102e-04 4.40416927479343300877445699370583e-01 1.81886663427438814884251616632582e-03
            3.15573901237963753280135748013890e-03 1.59007906197327880670755462233501e-01 3.69601415796714863337690815114911e-02
            4.75751078372788592518727313063209e-03 1.77353796757252879512378740400891e-01 7.88580680056353011675440711769625e-02
            5.44219680621846075424530297937054e-03 2.70066735820959358260751059788163e-01 6.88475294314979391341324799213908e-02
            7.92017614394921812270844441172812e-03 3.41391033021149870041455187674728e-01 1.15999807640960211907454890933877e-01
            5.32008534775439342889846727757686e-03 3.73937979719584367988005624283687e-01 4.83174342873769688266882837979210e-02
            1.27263581267450737524726811500386e-03 9.91330633416821899173143606276426e-02 7.12831450125742636936276852566152e-03
            8.95691044613802951623870995945254e-03 2.99506418629674531217688127071597e-01 2.03692910584251041239411961214500e-01
            1.63186984102462145744538712932581e-03 1.78629848603616248059466897757375e-01 7.23616174794815877197029507783554e-03
            2.72731918398721410318197300171050e-03 3.62068801895972014470714839262655e-01 1.29138832500325321378520371240484e-02
            2.63628096071471000258479300271119e-03 8.87929154893665473213815175768104e-02 3.76879497842590865941403421857103e-02
            6.87004129601127659643378109421974e-03 2.33622810141715209208967962695169e-01 1.37006694087071001497690758696990e-01
            3.65717045396642386617669373549688e-03 2.56595409709019783850436624561553e-01 2.45400602475243957700801189503181e-02
            8.46491817063666179658809607389003e-04 4.10688191117846435052030074075446e-02 7.18882826169304090663247919223977e-03
            7.55851039229440209606114553508860e-04 2.79416188649260710530342066704179e-01 8.91464317498128172405358338181713e-04
            ];
        
        S2=[
            6.84492577413612322823066591581664e-03 3.87642030404563353584990181843750e-01
            5.79363161800529671840997991694167e-03 2.11004508061496565129999680721085e-01
            9.00882035085073806790489925333532e-03 2.99492315804508413279449996480253e-01
            1.69864886095236829362253860864485e-03 3.72229259924406336024205188550695e-02
            5.74576293128239989904670892428840e-03 1.45109243574500224038814621962956e-01
            7.95565506872921053560521187364429e-03 4.24759304540574700315147538276506e-01
            6.82713759376400806250240194117396e-03 4.62208708748706120772453687095549e-01
            4.59141062991001887294562067154402e-03 9.29497017007697989843251207275898e-02
            4.03255144162308434320485250879074e-04 7.83534428260363076390593306541632e-03
            4.22204297326053863692596479495478e-03 4.89039369660395462080515471825493e-01
            ];
        
        S3=[];
        
        
    case  26
        
        % ALG. DEG.:   26
        % PTS CARD.:  130
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 2.0e-15
        
        S1=[
            6.99263224080136380246541794036830e-04 8.00716549403165389620085079513956e-02 4.79466097543667854607996048343921e-03
            6.02782386858443015344166671098947e-04 3.16436115715307691997892902691092e-02 2.91551962068358448687543926780563e-02
            1.65277235648383577504283437065169e-03 7.53800475153986598852995371089492e-02 2.62093640224986615394620059760200e-02
            5.42853671498377669292489944297131e-04 3.31000343360322718533339525492920e-02 5.69811791687521804888882925865801e-03
            3.20179894985641059698688870582828e-03 1.32486189614567290151470047021576e-01 4.17247227421209468389839969404420e-02
            2.30710553818916017099205362228531e-03 1.08687132914402101446782467064622e-01 1.00045659106527559667476623417315e-01
            7.18973661379937369758641452222037e-03 2.50272313290526460249907358956989e-01 1.20614402205249035548995095723512e-01
            4.12988360854342137495898157339980e-03 3.89022062042761740396912273354246e-01 2.95379425169078370372321984405062e-02
            6.86397910804285426439319550695473e-03 3.58509296427661550143994872996700e-01 8.73784651638445236665475590598362e-02
            5.19882276408716471199911168810104e-03 1.86869179476221558555693036396406e-01 7.63119015129594124857703718589619e-02
            9.28573735499042449657858266931498e-04 4.14705909590306287526573214563541e-01 2.05753096537086583445175236306568e-03
            8.79958359034760904948679893777808e-03 3.19415305383438707931276212548255e-01 1.70478728497248960271903683860728e-01
            1.48338083132825348728345282722785e-03 1.43737626199764018108950835994619e-01 7.99960809148430283810338181638144e-03
            5.05356221604434489858004297957450e-03 2.83788138859470406849538903770735e-01 5.11658736851377873855462041774445e-02
            3.13466892304028587473707645472132e-03 2.16546666473477117387247403712536e-01 2.27845992508956714239776175645602e-02
            2.29577919369932000048795117663758e-03 3.12898503074879985597078757564304e-01 9.47329791221356194708125286751965e-03
            5.69774457934106972126220291841037e-04 2.26434797407717519623915336524078e-01 4.64007732175652793521480887761754e-04
            ];
        
        S2=[
            2.45691265148301023202304094184001e-03 6.67371225764660652712834121302876e-02
            2.63476558340936049384256589789288e-04 6.34011649207668302741236487918286e-03
            2.65107959093367458527912283727801e-03 4.93753032896384780325860219818424e-01
            9.73403391859144380415802544348480e-03 3.88787497107593971179539948934689e-01
            9.76782346162377579368296665052185e-03 2.73147100929078678888117792666890e-01
            5.76425181732844946203009328655753e-03 4.71828563321166016830687794936239e-01
            6.62762972427263785668571927089943e-03 1.54201430364544173068352961308847e-01
            8.47217253926405011732470029528486e-03 2.12043163302205517339871221338399e-01
            8.20620030129395536255820786664117e-03 4.35985419384383177110464657744160e-01
            ];
        
        S3=[
            1.02433312946116263780771404867664e-02
            ];
        
        
    case  27
        
        % ALG. DEG.:   27
        % PTS CARD.:  141
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.1e-15
        
        S1=[
            2.76589741688336183261798240096141e-03 2.87042196593496612422313774004579e-01 3.07306047272728651009732203647218e-02
            6.27871810201825926445184933299970e-03 3.45087841715568388600843263702700e-01 1.29152640063449730689626449020579e-01
            3.19757634972719657212580912641897e-03 3.75930157048661783836251970569720e-01 2.80334860952500125597097735408170e-02
            6.85769661527540954487625057822697e-03 3.16945588933131960907019220030634e-01 2.09130921137668762677108702519035e-01
            4.93113505949477684553450274052011e-03 4.07228393042719849503896512032952e-01 6.60389128497386734828467069746694e-02
            3.22768645246484034427858489380014e-03 2.13553598457823928802312707375677e-01 4.10305768191818395762560101047711e-02
            1.46391318089954909284489126264361e-03 3.28852878068892628959929425036535e-01 5.29964037179903642987843781497759e-03
            3.56531765524350672744158252669422e-03 1.39295306142148717087891895971552e-01 6.30739954149509007841345464839833e-02
            6.17383156543067218580134181138419e-03 2.55246254696978036591303862223867e-01 1.48962850938240159992531630450685e-01
            5.34685029480812343966045219190164e-03 2.08376015600374053793331086126273e-01 9.46970824331307137189384093289846e-02
            1.62123379881966810101001641442053e-03 4.40010551946215466845302444198751e-01 5.58071701526011761801404631455625e-03
            5.46530554645663444596204172398757e-03 3.02220941227821116026319714364945e-01 7.50769024331962436802356819498527e-02
            1.00756156364484967744021837177115e-03 8.19468025835336788542662134204875e-02 6.98252932445901815239341914320903e-03
            5.98386804236580556806690545101901e-04 3.43649699121419890945894337619393e-02 6.09356940376483413590591453612433e-03
            2.16357901768035338491147356876354e-03 8.01120738471011173409408456791425e-02 3.50344225276973952887260566058103e-02
            2.31119355589055148689214824742066e-03 1.47213431898922469942547763821494e-01 1.93520013180389742479814429998441e-02
            1.69712686940351117903524880148325e-03 2.29719653257843214877453874578350e-01 7.33247254904045766416009755062078e-03
            4.23303067881924672925808561529948e-04 1.47655521119869792556755783152767e-01 4.90328443462974511261875942835786e-04
            ];
        
        S2=[
            4.78004248372995363403070712138287e-03 3.80714021181187134423851148312679e-01
            4.70507990472710576485004807523183e-03 4.46667803703864552122837494607666e-01
            6.02511351207520723383126437511237e-03 4.16141378805412132102503619535128e-01
            2.60631093640093436034077534202424e-03 8.03046477884382658984918634814676e-02
            6.73565769902467748053709328814875e-03 2.33400406669870996712035093878512e-01
            7.87398289068131675139117220396656e-03 3.01165465166509105365832965617301e-01
            5.64122127234918148419584582597963e-03 1.74779966354899979341652738185076e-01
            3.55861870643731570060253055487465e-03 4.85565054185162769950778738348163e-01
            1.38866976447708833156124974550494e-03 3.25715201801814874249529907501710e-02
            4.87162246140885810130605193535303e-03 1.27570901904677458382764143607346e-01
            2.87721202835250771008557268970662e-04 6.63921918095859882325004974745752e-03
            ];
        
        S3=[];
        
        
    case  28
        
        % ALG. DEG.:   28
        % PTS CARD.:  150
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.4e-15
        
        S1=[
            1.05876977884044850067302157015092e-03 4.55054005583463913597874750394112e-02 2.15243853694561254608075273608847e-02
            2.94783611725712079917882668667062e-03 2.13394454767087293767247047071578e-01 4.90696693575595060954341875003593e-02
            6.27862810833843508484530815394464e-03 2.42102511919319640654180147976149e-01 1.77658450296370312715410477721889e-01
            6.55305706204968758943341100575708e-03 3.27190732019169983058048956081620e-01 1.89812356292736877927396221821255e-01
            8.89447709627756103595297432207190e-04 1.41998166933174241322035413759295e-01 4.45838202328932217688439720859606e-03
            4.00596464334747846824935635368092e-03 1.75396393191461719140988861909136e-01 8.76779764843520492689776801853441e-02
            4.23234786713829955195897980502195e-03 3.92139613334414494083546287583886e-01 6.31803276344106712336667897034204e-02
            1.18838213384389081161973678035793e-03 3.34145615035921328406232078123139e-01 4.14946413392367364353363612394787e-03
            2.13847116750897096523753582175686e-03 1.74146196051182144959312836363097e-01 2.27948049259162482949037098478584e-02
            2.56563869101864667696122346285392e-03 2.75839008071824209888944778867881e-01 2.27008443717970145869777809366497e-02
            4.74270212944699902542222691437246e-04 2.62226671646522725700467759679668e-02 6.14964854266397079746164777702688e-03
            5.12299328085209695199386814579157e-03 2.43047202365926140466001470485935e-01 1.12033629342270985018714668512985e-01
            1.16227268223202434734020460638249e-03 2.29992984057907162709355475271877e-01 4.78148977298713338118796301046132e-03
            4.46255807013827207607725711113744e-03 2.95608280872401651961922652844805e-01 6.24487421796328864842529071665922e-02
            2.95894188617694311321804079284448e-03 1.21393450754091192234085383461206e-01 5.02111859134281163230539846153988e-02
            3.13026732148427404289758868571880e-03 3.73823800310209619635060107611935e-01 2.57279987428787430825405380119264e-02
            1.57008882174401688189280257290648e-03 4.42783406520243560944294358705520e-01 5.64656599346616185769454077103546e-03
            6.39356921147789681453277310652084e-03 3.37798663200582327537091487101861e-01 1.18089069715095071333443854655343e-01
            1.60245809539651638819801604540771e-03 9.22291891952822101563569390236808e-02 1.82422910122947215527666742218571e-02
            3.62567297493041269026287087839933e-04 7.02907404781327321829920151685656e-02 1.20025560148715230122939967571938e-03
            ];
        
        S2=[
            7.18123315032306169802822637393547e-03 3.03982922516484066477460146415979e-01
            1.55567604340748054944990053094500e-04 4.80412619665786157913123233242914e-03
            4.42585250544657703847262197882628e-03 4.58279904240411928295628740670509e-01
            7.10529319522408900344778714952554e-03 3.86267973570042055797557623009197e-01
            6.54637429454447885429635434206830e-03 2.58264072150462098420575784984976e-01
            3.90497168554316794256142664210074e-03 1.05895844178627504539313974873949e-01
            6.77379780166274987618368896846732e-03 4.29552202118899328908696588769089e-01
            3.84105528929750902888962116321636e-03 4.84841132562589427479338155535515e-01
            5.97133482012411013345465349289043e-03 1.58637688863059594934767915219709e-01
            2.56412600233903251872868978011866e-03 6.08391923927586036890957643663569e-02
            ];
        
        S3=[];
        
        
    case  29
        
        % ALG. DEG.:   29
        % PTS CARD.:  159
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.3e-15
        
        S1=[
            3.84626485738124961055728334713422e-04 5.89421088402292056573905654204282e-02 2.72874324792107044584943409404332e-03
            5.22610734846122495372844340977281e-03 3.49788010009331795657772090635262e-01 1.57177699867193487737182522323565e-01
            6.76411974626205288185776254294979e-04 3.23001823543550170558802392406506e-01 2.10096664482755952213510752812908e-03
            3.21956835376690760422424553155452e-03 1.58145854249516104594874832400819e-01 6.81658088137464396316644865692069e-02
            6.36097218585816512326225335982599e-04 2.95494682613533683757101044875526e-02 1.08309586036093533772950436855353e-02
            6.77262328600379635151762514055918e-03 2.91819173426537070703545850847149e-01 2.18932341980172556761274904602033e-01
            1.38076537479522409683496686483295e-03 7.55221785129957801085964774756576e-02 2.12868962407332593111952689923783e-02
            2.25634342274366826025522314580485e-03 1.17116669508898874063618222862715e-01 4.08472165761024560981518050084560e-02
            1.34746115929350795611951774333193e-04 1.11662181081691909423359021502620e-02 1.60349649604376361805568862450855e-03
            4.69316533814183832473698743115165e-03 2.08045649279087141447774911284796e-01 1.01545985226834029058196051664709e-01
            3.91145922915324493335242905800442e-03 3.92210114980434843001688705044216e-01 4.15270612688226833930649206649832e-02
            5.31519818159831465564924002364933e-03 3.59711275509975836328635523386765e-01 9.38490411451324418878883193428919e-02
            1.52561941167259768059594193800876e-03 2.45874699482875436196138707600767e-01 8.68602980438413878327974515514143e-03
            1.91268333586500863946477934973700e-03 1.67002738174923143210648390777351e-01 1.75891240440456259053192411556665e-02
            8.70847615650875879478309826708937e-04 1.15008598631946429291517119963828e-01 5.52352451221255549840272180972534e-03
            2.83457674083273337625743515388876e-03 3.15395398117319147868187201311230e-01 2.38589269426556054021215658167421e-02
            3.29099888192610793777315159047703e-03 2.23222650224820601261299657380732e-01 4.02953345447718061511643838912278e-02
            4.58946208246380476375447798886853e-03 2.88395859918732400117846736975480e-01 6.78784043114470936330917538725771e-02
            6.25918324941722187954029621437257e-03 2.67366350272775554763882155384636e-01 1.39535607181082654815185151164769e-01
            1.82067020564037119513545182059033e-03 4.05765395298891551334463656530716e-01 8.06658570416661502289201735038660e-03
            3.44336312520880964086616993213852e-04 1.86320727675359537611754490171734e-01 1.23446812287404965226972342051681e-04
            ];
        
        S2=[
            7.58231051578452173826672133571947e-04 4.98914824637686160624383546746685e-01
            5.58555014758393714863204948528619e-03 4.34380426761730609186429319379386e-01
            1.43024575307844565089099297949815e-03 4.10973356271179326992815106223134e-02
            6.26960172131162292441519667818284e-03 2.08405305132400797685221505162190e-01
            5.28570892911379552198125608697410e-03 1.60745884431963526051134749650373e-01
            3.06720058235945966171320797855060e-03 4.88401602932602707518583429191494e-01
            8.15511940387161383958503080293667e-03 3.02386411215128436680288359639235e-01
            4.08643922061357241920509508759096e-03 1.14426812994425419756083783795475e-01
            5.15655831762925741940106760807794e-03 4.64762431080738891431280990218511e-01
            2.80442356641565646371838660400044e-03 7.37218813900996500221296514610003e-02
            7.95660764204446369074474887383985e-03 3.90619178783263687471105640725000e-01
            ];
        
        S3=[];
        
        
    case  30
        
        % ALG. DEG.:   30
        % PTS CARD.:  171
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.6e-15
        
        S1=[
            2.10737923195621632346807849955894e-03 2.59053884521067367963098604377592e-01 4.78351231407725743793868389275303e-02
            2.96246137254600619762023860914724e-03 3.91572188291256340075818798140972e-01 7.96595269316006521220785430159594e-02
            2.97211009221224264562066785799743e-03 3.56140283296233539633846021388308e-01 5.76934012738742479919906713803357e-02
            3.43859219376796160921871958748852e-03 2.83012497349588776351936303399270e-01 7.72614375768841260549990579420410e-02
            2.04624948417359920141422868766767e-03 2.41613762451524383134682238960522e-01 2.27583842950000725391390687946114e-02
            4.43632725300867076373423358859327e-03 2.52781247187932933684351155534387e-01 1.23811978770674652183814146155783e-01
            3.76716146477324006699993752533828e-03 1.84906106383917107072178964699560e-01 1.15881967236100599794035304057616e-01
            3.40200337805093627405184086853751e-03 1.93704377153640139308521384009509e-01 6.65174447818816239053063554820255e-02
            6.03348284271058811142895983437029e-04 7.64012484393875451749167382331507e-02 4.43878137706136145085311639491010e-03
            9.79474288946621330156916407361223e-04 2.09048087452689629328617115788802e-01 4.66357939268862656950753731166515e-03
            1.13722295049933875762948609633440e-03 2.98542994059242672300058529799571e-01 4.70368176447704539056005046404607e-03
            2.68234242309315414759529616617328e-03 3.34379040034031072003983808826888e-01 2.51820667038687127925467024169848e-02
            2.93503490153276206525712055395161e-03 1.23230802380695114717212845789618e-01 6.57765738247428899398983048740774e-02
            5.65151177146874100776230420706270e-03 3.38514162429845721913324041452142e-01 1.26124094984989471024405816024228e-01
            7.15824263483379272937634496543069e-03 3.54403602180687449241247577447211e-01 1.94870950923518471142870112089440e-01
            6.46301172505914581223329662407195e-03 2.63068297757808267078871722333133e-01 1.91001424572283140523865085924626e-01
            3.14079829044583078156982480777515e-03 4.34566173964669621998524462469504e-01 2.75334061245498980252666143542228e-02
            2.35096849705298269542219635752645e-03 1.64070987069878326813920921267709e-01 2.80639219813729780261812862818260e-02
            9.15253438074413645277049855764062e-04 4.26819997060804007249501523801882e-02 1.59024162689347069166156956043778e-02
            1.91094027354750959149520816993117e-03 9.39597946527298694130436729210487e-02 2.72942306520957758686485306043323e-02
            9.59940278734458421298902930374197e-04 1.35408853519934452291906268328603e-01 5.69121144541610401307307753882014e-03
            1.32128396158162522526502424824457e-03 3.96221514739659108617075844449573e-01 5.16234701662132353106038351597817e-03
            1.67810855733201104086904642542777e-04 2.94840425976739400892867593029223e-02 5.33708660694491229814495625305426e-04
            ];
        
        S2=[
            8.58649506855246912647577905985941e-05 3.31872493664435528476497339056550e-03
            1.90096633420775806659619444616283e-03 7.23724072246778316053195112544927e-02
            1.42221683384374710662489960100174e-03 4.71579102421717660797995108623581e-02
            3.89206436616542240242155870078022e-03 4.68030173651125425138275204517413e-01
            4.23275072664444369351793850952959e-04 1.26866046744675987217876311774489e-02
            3.69345591745714281126100608787510e-03 1.21591508222727892607473165753618e-01
            5.37642548271640395574078041818211e-03 1.82409561517452856183041376425535e-01
            7.79804566291289556062560350824242e-03 3.62287279352949798116867441422073e-01
            6.20247951407300237430808564909057e-03 4.36743248548460194502496278801118e-01
            7.45727538134287760290819235819981e-03 2.72428040783928182566597797631402e-01
            1.39565905987038387306431452827837e-03 4.97319339000308557885432492184918e-01
            ];
        
        S3=[];
        
        
    case  31
        
        % ALG. DEG.:   31
        % PTS CARD.:  181
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.7e-15
        
        S1=[
            2.90314662418183176695318792326361e-03 1.39090628472094579581863627026905e-01 1.06669680864077562332248305665416e-01
            6.41925849384847617071225833740300e-04 1.97233130789653043901665796511224e-01 3.26115566372831879038418634308982e-03
            1.41325713796386842266428196523975e-03 2.38524150984310012724165517283836e-01 1.31398178404036562044732150411619e-02
            2.52754523436128967182889848430705e-04 2.97507293859899553389869453212668e-02 2.34422112868689367162766679086872e-03
            6.51347724486725314883839299540114e-04 3.95783933296885381825092053986737e-01 1.96140036989502072536928345414253e-03
            5.09628972217063443139739575826752e-03 3.51680058776080939342278952608467e-01 1.52636565775219751328606321294501e-01
            3.93343524786906630075211666053292e-03 2.02482194261758341458090626474586e-01 1.13425278347502905829102815005172e-01
            3.56040462070939369224054082962994e-03 4.22553955222215060771873140765820e-01 5.45213844446831366608030577936006e-02
            2.48205723825383308203673671243905e-03 1.24680508847443971554014296998503e-01 6.06049291607722168118144168147410e-02
            2.71821873650677494835270486817080e-03 2.55546236402857285252565588962170e-01 3.75889521575177576462323258965625e-02
            5.76291470042130369623445407256668e-04 7.45031299353863113443807719704637e-02 4.38633017006165564638520137918931e-03
            2.08916634591076394791842041342989e-03 4.16807275523650544446496724049211e-01 1.72734143249371197015307188848965e-02
            3.18121152643343653443341878528372e-03 1.92871723006198453331805353627715e-01 6.25780220143109106034629007808689e-02
            9.06384644141954715079734228311281e-04 4.73478189395524634575096456501342e-02 1.84079470312829319988967569088345e-02
            3.34497962252281073880499917549969e-03 3.41905005347527657377781906689052e-01 4.29032089427714424800797132775187e-02
            1.81985674247371891250335540490823e-03 3.25813953057509653188361653519678e-01 1.31890837790157033904137051649741e-02
            4.64685168846558696847254266160121e-04 2.89425948757137285749507782384171e-01 1.19688163070959881391364110214681e-03
            6.44779339399209269756418905217288e-03 2.98789795676013270053772430401295e-01 2.19132761430176847827766550835804e-01
            4.23794885856830678971807202515265e-03 2.74475747742794984329606222672737e-01 8.11298915693309669050492516362283e-02
            4.81113776584835874106094166791081e-03 3.53229873891415580011710062535712e-01 9.52781144820168263143145281901525e-02
            5.79983451569280727272426290141993e-03 2.66367882475071815484568560350453e-01 1.46888708002111900707831182444352e-01
            1.83329335246342167535515965681725e-03 9.60481521010510957969685819080041e-02 2.75306300785874945302733607377377e-02
            2.22266899562619923286233003523193e-03 1.67407824778356056905082027697063e-01 2.58751082199381692017237099889826e-02
            9.10626392061235573685551436540209e-04 1.30677633770288320613772725664603e-01 5.79731868473983870393428929901347e-03
            ];
        
        S2=[
            2.02199419825968128988646910215721e-03 4.86152467913059771476014248037245e-01
            4.42154132154415346889209104119800e-03 4.19588080864260470903559507860336e-01
            2.11186492555161383841144662198985e-03 7.92067965066985402167176744114840e-02
            4.93433072513541331333142281323489e-04 1.77313492497139200965161620615618e-02
            5.61850175400658343599635813347959e-03 2.18436496790928025291123049100861e-01
            4.77959798336420604858432170658489e-03 1.72450787183886011799671678090817e-01
            1.62954332509288447698725388335106e-03 5.10951808223812711795552843341284e-02
            1.16383032102821510830881202025466e-03 4.97748620626470295036369861918502e-01
            6.86969759828412723096091596630686e-03 3.85168469245680045176527528383303e-01
            6.17952025467344723957952012938222e-03 2.97491490367992827970056168851443e-01
            1.72200689636290441225999203744834e-04 5.08280946272773696825719724756709e-03
            5.21698864789028708688345403743369e-03 4.50340941739018174594377796893241e-01
            ];
        
        S3=[
            4.17156040694318264866469192497789e-03
            ];
        
        
    case  32
        
        % ALG. DEG.:   32
        % PTS CARD.:  193
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.7e-15
        
        S1=[
            4.17460956476897303984996412395958e-03 2.06831686315902013584278051894216e-01 1.58867941229046738538244198934990e-01
            1.52032612888650065993689786125742e-03 4.46761426533167027574933172218152e-01 1.47664424308546586572798986480848e-02
            2.64302180126624276443325634033954e-03 2.32466207646378680840015817921085e-01 5.14311272402330696418282229842589e-02
            1.24663496233453578027061681154919e-03 1.16834926706957942244002879306208e-01 1.95597403645923935411321537003460e-02
            1.51029052835681474979689919990733e-03 3.28240545844934006325388509139884e-01 1.26185667501572066490078682932108e-02
            1.82208778843809260372665903560119e-03 2.59350932003120149271779837363283e-01 2.10621566756392201991143764416847e-02
            3.74054484958008179318333930041263e-03 1.65463998237349529940587444798439e-01 1.17543509254347977588395224302076e-01
            2.28407706778600968697268136509138e-03 3.82464907380937246816898777979077e-01 2.92988505404852273006355289908242e-02
            8.42601820267144794619762482312808e-04 3.97817072646720837347800170391565e-01 3.44282165227435561077884429437290e-03
            4.10460332323965733369863784218978e-03 3.25417469646690105111730417775107e-01 9.03702811448642562286437396323890e-02
            2.23283842380996890020217193750796e-03 1.13339191487603074826040483458200e-01 5.06976359267404738662143870442378e-02
            3.10141736076142418218615226521706e-03 3.11041632827641745251412430661730e-01 4.79444789155617986931012808327068e-02
            4.44446271575963845149731534789339e-04 2.98837135388912988887000210525002e-01 1.27665738027984911225065633999520e-03
            7.42614664592584471786151922856334e-04 1.52112280943783789721734933664266e-01 4.26256341511160007085035061891176e-03
            5.56353172795318842887291044263520e-04 8.86617050339115198021033847908257e-02 3.85251198871905528239856231209615e-03
            5.98667955124139363615176634425552e-03 3.19152580593060508640235184429912e-01 2.01900148632476683641101544708363e-01
            2.94115803140993122000157455886438e-03 1.66755693582133324159855192192481e-01 6.77103932212691633774070965046121e-02
            4.22437173597556628323879124309315e-03 2.42137924019432793087958089017775e-01 9.55724171106545339338111944016418e-02
            5.10572795818304506904050654725324e-03 2.79861747997829701795069468062138e-01 1.50370864573458273838468812755309e-01
            1.24497957660826439815759947293827e-03 6.73984002072566534824460404706770e-02 2.43498506845871223736033073237195e-02
            4.94425282751114608670306438398256e-03 3.71687966693420357433552680959110e-01 1.29387867366923425826996663090540e-01
            3.70455657146825066694528771904515e-03 4.05628149065882848134378946269862e-01 6.32274204565012321044292775695794e-02
            9.72796620491534510914710676132700e-04 2.21980165737481027754540718888165e-01 5.26663302076577251320532724321311e-03
            2.13041829825008416018916612699741e-03 1.78325682037862948137174612384115e-01 2.61588819796510496651364974241005e-02
            5.27687184808808980827909973498890e-04 4.26221905914095089418225370536675e-02 5.55476787043224142642605656305932e-03
            8.50441238712440532527062075907054e-05 1.43443003792803787377341251385587e-02 6.23098020345069585589070637610121e-04
            ];
        
        S2=[
            2.66619511667080839195393543450407e-05 1.45744546236985564845178853232710e-03
            7.07313623831420435796035750541932e-04 4.98650893578847342890725258257589e-01
            2.79000757120663770913249202010320e-03 4.81168115995836753739922642125748e-01
            4.18929535206312604611600391990578e-03 4.52285756511600522955518499657046e-01
            6.33064728024302150743984896053007e-03 3.76328758047918987550417568854755e-01
            6.44136658858768772206726538342991e-03 2.83018279849573173301280348823639e-01
            5.44023301346713023668444364489005e-03 4.13679784325413357759515520228888e-01
            9.29639126596945933322935218257044e-04 3.02691910404441627546745507970627e-02
            4.00870525836537292679873090150977e-04 1.28101964801118073378205508561223e-02
            1.98817429746268219473148697318265e-03 6.20106498262642247265574724224280e-02
            6.26779886282444707867878008755724e-03 2.29277996928069882631717746335198e-01
            3.31469955752958017267117973858603e-03 1.01958826900269583504154979891609e-01
            ];
        
        S3=[
            6.51503549788129410202941471652593e-03
            ];
        
        
    case  33
        
        % ALG. DEG.:   33
        % PTS CARD.:  204
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.5e-15
        
        S1=[
            2.50541172828744806641565467941746e-03 2.58152928789506741491521779607865e-01 8.69286359632870375691027220454998e-02
            3.13333575729897798045620049833815e-03 2.84962401972082890466708704479970e-01 1.14092035322831353294503742290544e-01
            2.55304615093165584993606387342879e-03 1.34067633841934125138450895065034e-01 9.16653314400512203574677982942376e-02
            4.81962342942039104892737455543283e-03 3.26913750209671560842394910650910e-01 2.15939391243415657051230027718702e-01
            1.21061901943424741513732900699551e-03 1.00845715982775571695206906497333e-01 2.23903162865835385353374675787563e-02
            3.84867336352746761476972814364217e-03 3.55232825404763974397326364851324e-01 9.92419378880282920807331947798957e-02
            1.49412234597451603484996596193923e-03 1.58402734195724786969705633055128e-01 2.04436751441748401636466780928458e-02
            3.09942701297168110574631327835959e-03 4.01239748458054656943261306878412e-01 5.86121184566509023650127119253739e-02
            1.65849864695653674041830871743741e-03 2.27322013164966585385684538778150e-01 1.94249101655772464514182473749315e-02
            1.85446480629463230152542685402750e-03 1.04004000131427021269558963467716e-01 5.08701921937649848004525665601250e-02
            7.71323259235281544037021728144055e-04 2.51619019363036655612120284786215e-01 3.80625212881914110410463969458306e-03
            3.30608620553421092558288307827752e-03 1.94671527372046238024694275736692e-01 9.39537103695899794564283524778148e-02
            6.77589097875324849508393842967280e-04 1.75639631458033129707274611064349e-01 3.66697762401211810323609086026408e-03
            2.45181417177645728441848227419086e-03 1.65422503950626270663448735831480e-01 5.04257035060658373448738700517424e-02
            4.57883078590503808397854657030734e-04 6.04829240909602813824896827554767e-02 4.11214828094120989770976493105081e-03
            9.10908132213261376164403770872013e-04 3.32103565357523278223084162164014e-01 4.33667086499291924012178256475636e-03
            4.62118995085480498835606155694222e-03 3.79413216845638723295763838905259e-01 1.43764176240543023466145200472965e-01
            9.80315616846607045670181079799477e-04 5.65688972489456487480197210970800e-02 2.18658180283708390712060776195358e-02
            5.10495889319824772134559509595420e-03 2.90984916269330096127987417276017e-01 1.68307331342502364268653991530300e-01
            2.73188521868607779585769179675481e-03 2.36845394483151533426834589590726e-01 4.85139799052083930952505852474133e-02
            1.00555100489236291794181710201883e-03 4.14549152976443013862706266081659e-01 4.73723988093216474837277019105386e-03
            2.13416735131724067189851190562422e-03 3.07536188240941033011921490469831e-01 2.21352139073682022130107327484438e-02
            6.43938204447683324588291853274313e-04 1.10926664433470392734371046117303e-01 4.39339510946666188945242126351332e-03
            3.29994166362838729777973156842563e-03 3.18863122056367798151654824323487e-01 5.51602053562473687575717917752627e-02
            2.47371640440409391523135518298204e-03 3.95367759101382776343314162659226e-01 2.43998884696386975079462899884675e-02
            4.59074077896355105254944106718540e-03 2.13757830297353068615962001786102e-01 1.50634941829186158601672218537715e-01
            3.22000887173715859366357472381992e-04 2.47793702175406298227056112182254e-02 4.50238528829736278646933556046861e-03
            ];
        
        S2=[
            1.60118200620710301324001179068546e-03 8.66842399345696401624650206940714e-02
            3.41733996723404386680611466431401e-03 4.09427960802643320370464152802015e-01
            4.46708090406837139002549363908656e-03 3.78333538296242721798989805392921e-01
            5.20545618643210522905051540476506e-04 4.98979717860051186217873464556760e-01
            4.86942861834268551218807274949540e-03 2.59660668008721118305004438298056e-01
            2.73216284202301793598843815402688e-03 4.78511973256382172614564751711441e-01
            6.90684531513952103091991130412453e-04 2.35174430579130483431882225886511e-02
            6.04436999045828132065549809226468e-03 3.06023861760360560424487630371004e-01
            3.68872324836709520809452378387050e-03 1.46489050716642699168090757666505e-01
            1.60467522197081007354069193127089e-03 5.43792032794962812425332288057689e-02
            4.96318808977210066540886046482228e-03 2.18192015863253874208282923063962e-01
            1.93118101498302059682021791786610e-03 4.91991344456553236774709603196243e-01
            4.66689888482328934377063589522550e-03 4.53399961617805269398218115384225e-01
            1.46741366789008735116786596996974e-04 4.72666051056800425728132708513840e-03
            ];
        
        S3=[];
        
        
    case  34
        
        % ALG. DEG.:   34
        % PTS CARD.:  214
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.9e-15
        
        S1=[
            5.08429861835657845954214439920804e-05 3.85126251838280975192674304707907e-01 2.10967602701019364686008117359961e-02
            3.27023983293754161912753053442771e-04 5.28200673454987598698373574279685e-02 3.38811215039004848606496445029279e-03
            7.59905829130820813926971979412883e-04 5.08987652831316070289879860411020e-02 1.82083287994723903857607893996828e-02
            2.33481884197296639740781820648863e-03 1.29124372072608517569847208505962e-01 6.05107706433122560785164978369721e-02
            1.47605781528109243974644648744743e-03 8.78179027775244785969377403489489e-02 3.59198913635777883346378303031088e-02
            4.93046515645237121844290939520761e-03 2.73553738024819192986569760250859e-01 1.64061028670212660784955005510710e-01
            8.63136629420850967631118066947238e-04 9.44770656323741658821901978626556e-02 1.06832635235884227525859913043860e-02
            2.51809125066697380226909963596427e-03 4.11152233303600511682418527925620e-01 3.11326803063737105048414122165923e-02
            4.82894011472341164181720785109064e-03 3.59447569498255292685939821240027e-01 1.33192623221705830349748111984809e-01
            3.80595751044294529352729483662188e-03 3.88410085330437082618004751566332e-01 7.42973192372942126127099982113577e-02
            5.53293194379957369716249715452250e-03 3.25355043342057725919147515014629e-01 2.05166497814761955575946217322780e-01
            3.25191985632487887830643913389395e-03 1.74780479481068395930520864567370e-01 9.02196061251755165866228480808786e-02
            2.41634362674254983549548825294551e-03 1.94164618149563289462022908082872e-01 4.39158729045114326616783273493638e-02
            3.29675723387472729628000500667895e-03 2.48327873404578958194122151326155e-01 6.89638299646016944555171335196064e-02
            4.14263325655033799732596122566974e-03 2.22992041261200868440539579751203e-01 1.25181798142886657743133582698647e-01
            6.99714692701545039159316541343969e-04 1.49604423548951093181358373840339e-01 4.67029856128329395881237928733754e-03
            3.04763201624417764956653797980834e-03 3.28391413692126599155329813584103e-01 4.89259225038644121608122361521964e-02
            1.57160307407964799052135784762640e-03 1.42385500324542807470962202387454e-01 2.46582497782073459258178615982615e-02
            1.11566589667665290973308511723872e-03 4.23840299720368851588148118025856e-01 5.95994324974427527180642627513407e-03
            2.21786548672098281567710564843310e-03 2.67690419344305596816013803618262e-01 2.85187228422853940312808873613903e-02
            1.74416367287191808634061462868203e-03 3.45411378952901138017494986343081e-01 1.55626433010154947844672079781958e-02
            3.65922936835056549810529791244562e-04 3.53449519657663746574627339214203e-01 9.25718233933834830362563650396623e-04
            4.12539871537648435151357517725046e-03 3.03335639471789153542857775391894e-01 9.92862116185854320926651439549460e-02
            2.48983786598559492676124849097619e-04 2.17555603201839350968604946956475e-02 3.97450587251847731184994927389198e-03
            1.40290906865686636033341283535947e-03 2.07546868273520451753100246605754e-01 1.37796394053688077419028701342540e-02
            2.88825287117379879680417742449094e-04 2.13560239517947836329625488360762e-01 7.72139120327043496430641233985170e-04
            9.81314402741670583144029116340334e-04 2.79386431782223720876601191775990e-01 5.46179089429344854361714212132028e-03
            1.43672677304067366926484949907206e-04 9.68669225755471735572754710119625e-02 4.70969101854539203629197718914412e-06
            ];
        
        S2=[
            1.14948328440455651774931844499861e-03 4.59825951541271496503249238685385e-02
            5.49840283231600073302058895308164e-04 2.08387475149684059383403678111790e-02
            1.99868672682467635182135268223647e-03 7.57621741867907788625657872216834e-02
            5.87000329562880392847423749458358e-03 3.76601980517545675830604068323737e-01
            5.88060381970682746061163115314230e-03 2.87355839822164893515576977733872e-01
            3.24211260691926756588587821283909e-03 4.74412498902845591874921638009255e-01
            2.91671683977711912461039389654616e-03 1.11172701285845609842084513729787e-01
            1.86468547170909088213364146469075e-03 4.91864961957689694749262798723066e-01
            3.84365534298185002742842542033941e-03 1.50887722762872905635234133114864e-01
            1.13480678385471947197150888175798e-04 4.15795739231892506199139702971479e-03
            5.40684965813482618091079245914443e-03 2.40434910413816760321736865080311e-01
            5.32372242124447888483862456610041e-03 4.15511596744837174988163042144151e-01
            3.80463083109051425963892389958687e-04 4.99542182661511791064157250730204e-01
            4.43046965007653018031597014214640e-03 4.48564265878708234858152081869775e-01
            4.70007051859918853486952627918072e-03 1.94386543578966120016104923706735e-01
            ];
        
        S3=[
            6.05048246508557534262262578295122e-03
            ];
        
        
    case  35
        
        % ALG. DEG.:   35
        % PTS CARD.:  228
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.7e-15
        
        S1=[
            2.27099691187698793370941530156415e-04 5.50454796915316388328243135674711e-02 2.06822021540594260316514585440473e-03
            9.85874665492085565723923146208563e-04 4.45376464915297476387934239028255e-01 8.90499418612781569459802000210402e-03
            3.91333152719676913202512258393995e-04 1.52384002524315248505715203464206e-01 2.11597756303986755596624469433209e-03
            6.32346551198585573975996521056686e-04 3.97516610028231298645806646163692e-01 3.26127342913124992906936938652507e-03
            1.77862813045475700278463815351415e-03 1.56078769403476091959248606144683e-01 4.51678948131286547851281909515819e-02
            9.21416331587753337228896555188840e-04 1.97295676276558723527543293130293e-01 8.60602170575682987041510330072924e-03
            6.64489967827079641704557388948160e-04 5.00938368999261987579707522399985e-02 1.39743532515683617500901192443052e-02
            4.51170642755311431665310450966899e-03 2.98859063014319081830194591020700e-01 1.87471332499894505208004602536676e-01
            4.30275224704192681135506148848435e-03 3.34763343265331358988845522617339e-01 2.32722344075948522901242654370435e-01
            1.80523399849691758116410067458446e-03 2.09372824976375354610880208383605e-01 2.90018846146845241107392610047100e-02
            1.76178038360539951541483549846134e-03 1.06564744861383425833523119763413e-01 5.35674828677535283527610943110631e-02
            6.89038261003183365865920606552208e-04 3.21362627669543321484724174297298e-01 3.49619434914352376561619273331871e-03
            3.47514792975928566570731348406298e-04 2.41587026347987537944206337670039e-01 1.17604279684175661371825505341349e-03
            4.06486029903899842574688250351755e-03 3.33153099720764844793308157022693e-01 1.13953928744148377805167626775074e-01
            2.74482817993335591333359246846157e-03 1.55116554813303442195149273175048e-01 8.49385995115550379574997919007728e-02
            4.11799366142912796201480674085360e-03 3.71014151677461878620079005486332e-01 1.52565186347216358297984584169171e-01
            1.38988675968128479187857493570846e-03 2.70883661807772069263222647350631e-01 1.36186781090766283186521690140580e-02
            3.16589094074170044820770719695702e-03 3.52506748415466475421453651506454e-01 5.62263874435523919315116359030071e-02
            3.47285535997079531159159593300956e-03 4.05720132778190467615075931462343e-01 8.17534894396882150235938979676575e-02
            2.78297055784304742456769510283721e-03 2.18949681014998343009381187584950e-01 6.36424301897473843059316322978702e-02
            3.66880507838134115733419449156827e-03 2.06709248085564512020795291391551e-01 1.15394220836074104519219929443352e-01
            4.35571331178043631010554648241850e-03 2.56601087879092248922319186021923e-01 1.48187116283310715836307736026356e-01
            1.36837471636101011725206166858015e-03 1.38206295211249968746258787177794e-01 1.85978710613813984220410446823735e-02
            2.35082879046609287410074529134363e-03 4.25810208585481020371332760987571e-01 3.29710199269463780091804494531971e-02
            3.58750447996635966549372476208646e-03 2.79725926952240189216070120892255e-01 8.53795515729538628812989031757752e-02
            2.36623486900244665333270521756504e-04 2.24666835802111858333418581423757e-02 3.63305584293396573516976566509129e-03
            1.26733105568389110492932836393720e-03 8.38622585244554186179044563687057e-02 2.60200776623639613849281460034035e-02
            2.53507865000570037708471637927232e-03 2.84800346414468275035858368937625e-01 3.94569261564381226770059640784893e-02
            2.02703604344201329667818356483622e-03 3.56594103536573836521483826800250e-01 1.97707177044081881811532497295048e-02
            6.27239193620508653573275559267586e-04 9.74242311151380380174202855414478e-02 5.02869741562738704898638530949029e-03
            ];
        
        S2=[
            2.87396701483509263083671037009026e-03 3.01435168617317950090495060067042e-01
            1.14818404604971570012739157107262e-03 4.91641403982890690116391851915978e-01
            3.99767542093215806631167907880808e-03 3.57793135798597061736359137285035e-01
            4.39509294666390066402827940805764e-03 2.62413960109345423266091756886453e-01
            4.16018330616281430817116060438821e-03 4.02816450585781860027623224596027e-01
            2.63759414258074760539773961909304e-03 4.73088965500664149210763298469828e-01
            1.48410331952580172229505528491700e-03 6.28680490440657047113148792050197e-02
            3.80259126526817994862206262496329e-03 4.41597433754930324134591046458809e-01
            5.06560516369039624123837839420048e-04 1.96271599834707231868602406166247e-02
            1.03241781329916357604070675790808e-03 4.02868688301933966511292339873762e-02
            2.50181980777866748433257981787392e-03 9.85261546687923595033709034396452e-02
            3.35953990620869785352353709129147e-03 1.39534462989307272984262908721576e-01
            4.58095447046865354712164020156706e-03 2.21536010509691960734812710143160e-01
            4.18595331850648341015519449115345e-03 1.80750322135299568149235938108177e-01
            1.20166868702420654227425167448473e-04 4.27720726848295623723217318001844e-03
            3.13792170535282584787478699084318e-04 4.99913222486225106155899311488611e-01
            ];
        
        S3=[];
        
        
    case  36
        
        % ALG. DEG.:   36
        % PTS CARD.:  243
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.2e-15
        
        S1=[
            1.53084158353408998805289531475182e-03 2.55002336075419500094341174190049e-01 3.22922639737693653594341469670326e-02
            2.46163598764502326854408664758012e-03 2.71121513603339969744610016277875e-01 9.41402098044081675487504412558337e-02
            1.66186042169761630932223184231589e-03 9.26886044848755363823045172466664e-02 6.49936721042402160142259504027606e-02
            2.37136824651687722412263248372710e-03 3.29454376584550512330196170296404e-01 7.37136644534187868149643918513902e-02
            2.05690064632487422391360354367862e-03 2.88378841130442042928194723572233e-01 5.00649881468919338844969502133608e-02
            2.61767126144689214750638228679236e-03 3.92172416405173862585797905921936e-01 6.58963384748946778834977067162981e-02
            6.51699642263251258892525763499748e-05 7.62225001558761768721828033790189e-03 1.23411852776327143031698252428896e-03
            2.61385214196917974385780958357373e-03 1.78207936627975682686297886903048e-01 1.05695827327609861989721196096070e-01
            3.31556877077426696842632480866087e-03 3.10248931459913612762591128557688e-01 1.23871670395550900756198586805112e-01
            3.47617319858536943932847407268127e-04 8.29157021331251853357713343939395e-02 2.80739761856614068735726874592729e-03
            3.45918624369627491013656772622653e-03 3.79638679346943119785606768346042e-01 1.10689778326021240340004681002029e-01
            5.08853424677457996314067223408983e-04 1.98068292114543353932276659179479e-01 2.75642261370565543054933499433901e-03
            2.26549564853226800126284601333282e-03 1.47947945322073032281551263622532e-01 7.20516990960159003920537656995293e-02
            1.18964024781928446282408540213282e-03 2.29202766553467329746496261577704e-01 1.44286115706842923978037163124100e-02
            6.08439554852720416479316956070988e-04 3.57473850344384502797367986204335e-01 2.74256669741186880767669364900030e-03
            1.55651362018915462651080794387326e-03 1.25249672229437725157197291991906e-01 3.90243298422159254346119894307776e-02
            3.24092666406562745010089265917941e-04 2.09805646004062387732780337046279e-02 7.47466346593332695708022939129478e-03
            4.17258237302728287460018563592712e-03 2.69131611744009735520677395470557e-01 1.75571778601519262741348370582273e-01
            4.45190142462590591337923129344745e-03 3.46219872421893559710781573812710e-01 1.71226758850752919238757954190078e-01
            2.38190698469414006976929698566892e-03 4.42387742658068516821145976791740e-01 3.43061434540936835424496109681058e-02
            6.16571745718724295408363911263905e-04 2.74633883722658844739328287687385e-01 3.02963551184710737362859589438813e-03
            2.64051811768303834604409985331586e-03 2.23701284655144105872892623665393e-01 7.24869584556441376443913782168238e-02
            3.60197908152548942894988215357444e-03 2.26472947511629774108854462610907e-01 1.33817023340663948660633764120576e-01
            1.27502323317565230929859776409785e-03 1.63361966829370430787093937397003e-01 1.76742202382502745783288133907263e-02
            2.04656057870132969747634099633160e-03 1.90791272854378857193324847685290e-01 4.22082170395821076880338296177797e-02
            2.23469587180631862680102450724462e-03 3.58165870631430682813345356407808e-01 3.44163203969896150380414212577307e-02
            1.05061121151762385146632983179416e-03 9.93849892050521904751647639386647e-02 1.74529902610205088497874470476745e-02
            4.11044553657110233423066514291122e-04 4.50232897891652383837168827085407e-01 1.22783484472307590634754248526406e-03
            5.10667280776456837532162325032914e-03 3.04860292986679271010075353842694e-01 2.36263190066308664816219220483617e-01
            1.52011067741504796516083253266061e-03 3.13583606270695147699711924360599e-01 1.55229285287317442243448795125005e-02
            5.54805076808530359296078859898671e-04 1.34806288266682894327885833263281e-01 3.84804577084505751438658904817203e-03
            1.33345058300172623552692918735829e-03 6.58281924328198569940084894369647e-02 3.80724693230130931120847037618660e-02
            6.67804382032510367855904220846242e-04 5.31928439194577426518684148959437e-02 1.21702541539402027404026540580162e-02
            1.33088911946721083967223875355046e-04 4.06479626359585802242158081298840e-02 8.58559044683948331504186324281136e-04
            1.44384349566076090887190019884656e-03 4.06447567212284865956206658665906e-01 1.23358378103272329356920877785342e-02
            ];
        
        S2=[
            3.83524725895431498404186676509653e-03 4.12357340313630305050907054464915e-01
            3.43814785469807523976548679911502e-03 4.39048383245843498734473087097285e-01
            4.03399885831912021622036590429161e-03 2.35052325458213695608122861813172e-01
            3.08553034072575832827411979053522e-03 1.50821734779891525324302392618847e-01
            3.33061681062377058434864629532512e-03 4.64273777634669981839721231153817e-01
            4.04661253784454125193015272543562e-03 1.91161361877308921819462739222217e-01
            7.83711855575436756177198915906956e-04 2.93500490934269150922197866293573e-02
            5.50413808085681126314181454972640e-03 3.08145218991093261529101710038958e-01
            2.71177446206628510252945041258954e-03 1.11178264788380914973586754967982e-01
            5.25317300340855369150672160571958e-03 3.81002349603441869962949795080931e-01
            1.44795788173428295955713629439288e-03 4.94480996747700407567549518716987e-01
            ];
        
        S3=[];
        
        
    case  37
        
        % ALG. DEG.:   37
        % PTS CARD.:  252
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.0e-15
        
        S1=[
            1.04959604867473641279240759160984e-03 3.16024096496639927966754157750984e-01 3.38290906827213161411904707165377e-02
            9.61312586091038630796423714031107e-04 1.05359411960281923459348263349966e-01 3.31459895561884293480048313540465e-02
            2.40221505000938299889678972931506e-04 7.49810217841138243466403423553857e-02 2.03347607611732879295818321452316e-03
            1.51627373004474519900541817918338e-03 1.19486593380355193838582295029482e-01 5.55104031624588586946167367841554e-02
            3.65945316462084778083574398976907e-03 3.19235165045989144338989262905670e-01 2.38060847117024854124878174843616e-01
            2.69415942149049073700561862665381e-03 3.91788257723670096943635599018307e-01 7.52705489375098402149433240992948e-02
            2.28460795949074046279347882659749e-03 3.17543407214198269983285172202159e-01 5.36208316291131165121086610270140e-02
            2.86017995322923775092238685147095e-03 3.26764524592731753216412471374497e-01 9.12947670950428336045234800621984e-02
            3.35600242454283433785056089959653e-03 3.18749666827180144768050240600132e-01 1.35115749070284674182573780854000e-01
            1.15853875465657353213688729454134e-03 6.74578790813868262032926281790424e-02 4.07701237566786764410231569399912e-02
            1.54334630985266183086576652350175e-04 1.83794409210811804378238321078243e-02 2.99027787656391775758524254058557e-03
            6.02351384720529840766101159488244e-04 4.76569358622159489069503024438745e-02 1.72509604538075254909212929987916e-02
            4.08187312607179338119767209036581e-03 3.35722787038010062055093385424698e-01 1.81116409195531985476534941881255e-01
            2.18226201409495385377668164039733e-03 1.35671607916117625824981018922699e-01 9.01302718482900888652409321366576e-02
            2.74898275323659144718513314131769e-04 4.17693614359697928950510004142416e-02 4.01709708723125332591186165132058e-03
            2.62321907934608610987736732056419e-03 1.91263737438076231178740727045806e-01 8.09653202292540813411747535610630e-02
            1.62924472819564567123429288386660e-03 4.48920329956642827351487312625977e-01 1.68814585101307729209096919475996e-02
            2.39061749419019597268087906627443e-03 3.97083379134988478931234112678794e-01 4.14766785336368876135892946876993e-02
            8.04385951443429810317486960968836e-04 8.62263440628871780102571165116387e-02 1.37345861622121794692885643485170e-02
            1.49128325264966500943164540160524e-03 2.81487959942281984471179612228298e-01 1.79186043302918159014769372561204e-02
            3.04697442177643022451216658907924e-03 1.92376834744259289644574550948164e-01 1.28471390056792922340633822386735e-01
            3.54367037302832097248050935434094e-03 2.51794247517433122940389012001106e-01 1.37321159731595615616228656108433e-01
            1.57487631404059432067843449232214e-03 3.65034794936327955383603693917394e-01 1.74418264943838295655265113737187e-02
            3.12593497656047866750661512469378e-03 2.57156635550253476196758128935471e-01 8.61981702781003389368663647474023e-02
            4.74652661273614311782109709270117e-04 1.24181611934111021944282526874304e-01 3.41126780083469841389764987127364e-03
            2.01562604214640134908664848012450e-03 1.71650727809369157972341213280743e-01 4.45555167117895126183135801056778e-02
            4.08610877411519791274630009070279e-03 3.97723109198285573739894971367903e-01 1.22097548884562687177002260341396e-01
            4.47932951436091734975786238237561e-03 2.57896645669856483529258639464388e-01 1.95710371236428148922215086713550e-01
            1.23104937074616067015087050862121e-03 1.44150074481193601894091216308880e-01 1.83113919798339880307125326908135e-02
            7.10646460660963909250487091640025e-04 4.12259561001791008383321468500071e-01 3.26225803520382127975718589141252e-03
            6.52578247959340247176529548056578e-04 2.52428464791436324432538640394341e-01 3.47544150731571212936676928961788e-03
            7.22954763894363499308892695438544e-04 3.29366927485774507733395921604824e-01 3.53879008258183198887270037857888e-03
            1.46729752903975970215744162317151e-03 2.09581481700515409594132165693736e-01 1.88449390557523001332729251089404e-02
            5.98229502634915386548308813274843e-04 1.83640169446232209837788218464993e-01 3.63810210869633648272936099488106e-03
            2.43675220996146480942345924347592e-03 2.41696232469337524140229334079777e-01 4.57532869613174186795845344022382e-02
            ];
        
        S2=[
            7.56994885929007437982973804935227e-04 1.02303932157760305021731994656875e-01
            5.52341834522307190830092693545339e-04 3.50202358754684098252063506606646e-02
            3.79254617818526162248415900535292e-03 2.59243490456549541267605718530831e-01
            2.62503007716438814533876922041600e-03 1.39441235872292679243145130385528e-01
            2.94041733963891363870524209289670e-03 4.61168458890810040173846573452465e-01
            4.32227828704266538867306834958981e-03 3.79153717955037827813669082388515e-01
            4.38793626075407268080641465246572e-04 1.73411753639024328321749379711036e-02
            1.79311680137484190031282871302665e-03 7.79889301806426238350411495048320e-02
            2.49962141423790877928223785886530e-03 4.79244935556920326202856585950940e-01
            4.53942125276982338250020276859686e-03 4.12061405479689502051598992693471e-01
            7.07839054597255573630931735351624e-04 4.98410574413956974915151931782020e-01
            5.10331832458679485881036796968147e-03 3.09125948685350482314504461101023e-01
            8.66476678904295794002646258213929e-05 3.67156682198311505940790944180208e-03
            4.14530662852685954228304510138514e-03 1.89364261200196171142096090989071e-01
            ];
        
        S3=[];
        
        
    case  38
        
        % ALG. DEG.:   38
        % PTS CARD.:  267
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.8e-15
        
        S1=[
            1.24250261557132582988793867428967e-03 2.83765514701740717562472582358168e-01 5.22971139020114439421860197398928e-02
            1.84824458035341416921315538246517e-03 1.28516564969693936060224359607673e-01 9.90797627018412813137970829302503e-02
            1.73852199496849705705886490747503e-04 4.13075041428614328498092334029934e-02 2.06440514930518719677321470840070e-03
            1.03691343697038932704368452419885e-03 8.92719800534144680703718677250436e-02 3.67301756493483427812307695603522e-02
            3.68552054500562598776669531730477e-03 3.37205292317987237460386040766025e-01 2.25945824487785718037713422745583e-01
            1.56269905721992657249286295950697e-03 2.73851374124178703617360497446498e-01 3.27011819900537709093768512502720e-02
            4.53088107226028440240039873998512e-04 3.99162626294194411435967140278080e-02 1.30695090443300153404404539969619e-02
            2.49781388487334667439232482877287e-03 3.08106832759384330344687441538554e-01 7.37176145710159158586094463316840e-02
            4.62834099424956190577118420037550e-04 2.19624713138435623260846796256374e-01 2.70388221454352780867558259103589e-03
            4.14108323134647609425895709023280e-03 2.73464607114281443855929865094367e-01 2.19158263446028789278585691135959e-01
            7.68421232903655489036853776241287e-04 6.86830028555526772571226956642931e-02 2.07228118815140395503338055505083e-02
            2.61137036940718670827155101221706e-03 1.83013599840179452815647209718009e-01 1.00621720892076091380396007934905e-01
            2.35919639420995507980705063744153e-03 2.28799483229925271210447590419790e-01 6.94898178335092570057085481494141e-02
            1.80105850247642843356410402577694e-03 1.01882711486939128708861801442254e-01 6.83773554225138580076048810951761e-02
            1.18514523369667602176180665196625e-03 2.43563673395817054112555410938512e-01 1.43204413633164825275789766578782e-02
            1.80524491668789866839761604211390e-03 2.05887716517901192547768118856766e-01 3.89678567422942698228460756126879e-02
            3.27022726537275166755480704239289e-03 2.01869508887603077429417908206233e-01 1.49396461917944173292127629792958e-01
            2.03961074015630156042178100506135e-03 1.59116021010545172220673748597619e-01 6.22282118189532995611301657845615e-02
            1.16281945905291768904310512056099e-03 1.77947635855591812514830962754786e-01 1.76192414704859877383480437629260e-02
            5.02872011041631604248702824833117e-04 1.62036747949692083725636848612339e-01 3.94772766521245526555317084671515e-03
            5.71580885705873253423148216967320e-04 2.91854983858524597017947144195205e-01 2.93117988617291056266389581708154e-03
            5.94230661254127579505579515029012e-04 3.71878691749589151971377987138112e-01 2.77798098243856299988108204956916e-03
            1.45260947260546073069031880464763e-03 1.34969372652465280371103517609299e-01 3.43221858358502412977841800056922e-02
            3.16335759879328654378682905701226e-03 2.52453429906421944473038365686079e-01 1.08724039341924899604308052403212e-01
            1.36284311036006036688128517653240e-03 3.24235164108304496632939617484226e-01 1.49491079080454249833342927900048e-02
            2.26079579990258157720650622479752e-03 4.37416190646625679949721643424709e-01 3.73240771592634676401267768142134e-02
            6.44587138714711118010758372776081e-04 4.55998803661209317628077997142100e-01 2.96327193163695200642249183431431e-03
            4.08341959086390329358023931760613e-04 7.54872857005682990072514826351835e-02 4.20297816077350953195557181629738e-03
            3.47388966177611087718379856426054e-03 3.26054126652946585362968789922888e-01 1.14274187380454431139931159577827e-01
            3.54930896974515889055745709868006e-03 4.01199588505488202727633506583516e-01 1.12888085312921521152773607354902e-01
            3.84596017428895809581179676683860e-03 2.68760660341264001615968481928576e-01 1.60196677899755890450705919647589e-01
            1.46260950936205245086207948901347e-03 4.06014702573877173108485294505954e-01 1.51551286402894190563639398305895e-02
            3.02834849824288633579616458746386e-03 3.83510346834340154487108520697802e-01 7.00421815004378944813367979804752e-02
            1.66852816559418681450696619883445e-04 1.64470646369431552913376037849957e-02 3.44298850158080132349835977834118e-03
            2.26473446047441642758890623099433e-03 3.54297786162694750000667909262120e-01 3.76676144786773081651887196130701e-02
            4.20126870715988726123768515208212e-03 3.41995175798316008997090875709546e-01 1.65998426676715687966279233478417e-01
            9.05485926201706186472462078995704e-04 1.15288891079278379980799229542754e-01 1.20325311111651837847613677467962e-02
            1.46142831591413282013580676732545e-04 1.16528101812402543058233561623638e-01 2.98221681520129382886308277988974e-04
            ];
        
        S2=[
            1.58198222014196152387932592375819e-03 4.59072136031764554964240687695565e-01
            2.93082910955432202407267006094571e-03 3.93268006255360069722826210636413e-01
            1.96510845185057228187708311395454e-03 4.68182589206935673153964216908207e-01
            3.49734883664800603325639727358976e-04 1.75045040462672892023210380330056e-02
            2.60121125552193084573548986782043e-03 1.45480114652860703650105733686360e-01
            6.20233313265518054670927261540214e-05 3.04717426636049414090390285991816e-03
            1.25319110825724373478595730091456e-03 5.87361601120219983807047015034186e-02
            8.53775101957864220036942271008229e-04 3.73835575017464616376372532613459e-02
            4.49280475940249442862173268053994e-03 2.85504778291264027512141865372541e-01
            3.96885213756022810910328857403329e-03 4.20333501770014217324700211975141e-01
            3.87427659633373874803474734562769e-03 2.07689421653862921202104985241021e-01
            1.51794877726721681894528348522044e-03 4.92196272955310765162550978857325e-01
            4.98799680519128318945121236538398e-03 3.56565593983026118962698092218488e-01
            ];
        
        S3=[];
        
        
    case  39
        
        % ALG. DEG.:   39
        % PTS CARD.:  282
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.6e-15
        
        S1=[
            3.91201516419781921037046590328146e-04 2.38275208584557002255976954074868e-01 4.37448243826125699601492513579615e-03
            2.71296293947622833495070260312332e-03 5.02290201314918416386490207514726e-01 2.31437075889000587469723768663243e-01
            3.21678249161045553064508517593367e-03 3.50682734157158282339850075004506e-01 2.36099447436519921383890618926671e-01
            3.82255318815834753670007684078769e-04 2.77808917719284009795188694624812e-01 2.66233974789171091296369198175853e-03
            3.12974434051319170407845504655597e-03 3.21203482682616781040962905535707e-01 2.07439611934945505877792015780869e-01
            1.08104370344727293562492320688762e-03 4.57424216852982068459709807939362e-01 1.19091524767886494662327834248572e-02
            3.45426989161139847382686607346614e-04 1.91038342852184600539899861360027e-01 2.15696233889989567519829272157494e-03
            2.19054804285410704284275951181371e-03 2.06351940732836303116215503905551e-01 1.07257493751645838209185512823751e-01
            2.44886953539576006366740301700702e-03 2.61238382787471912216403779893881e-01 1.12438393291482544023018874668196e-01
            2.21549482544702641337308612889956e-03 1.59388899623120305903967164340429e-01 1.14090324996719685857193837819068e-01
            2.34993921431397032387189405255867e-03 3.98869513224066885026530826507951e-01 6.19448419160624624835165263903036e-02
            2.84747357948343486622189146828532e-03 3.12272579273360884055676933712675e-01 1.21962883225284618449180129573506e-01
            1.86161499437441217176536500943484e-03 3.64436262358476370515347753098467e-01 3.86447963475402486710486016363575e-02
            1.13555331101312202354836067286215e-03 2.30564834240443161661460180766881e-01 1.72339717685665617941115357325543e-02
            4.75092827573918936780866806302015e-04 3.92682833818939619474619462380360e-02 1.45623703030592060309045265853456e-02
            3.71655031528527347167711880615570e-03 3.58257733547722723166373270942131e-01 1.54161109997349499156626961848815e-01
            3.09321526687093969684716476820086e-03 3.80507882548312525994305133281159e-01 1.01140130678948808595762898221437e-01
            1.91871114087192569733208991067386e-03 1.31886575586608933985388603105093e-01 7.32858166481221984245308931349427e-02
            9.47936482876536525184196779747481e-04 1.72630255068886018232632295621443e-01 1.35174633972146174959449993480121e-02
            1.15793858431617623962195384734741e-03 7.07756358900085702501314699475188e-02 4.31486463812221776592537025862839e-02
            2.44135381101898417703296995107110e-04 4.73824731204699201025931643016520e-02 2.84833883835317481364457492531983e-03
            1.96662350376744640353954274303305e-03 2.97193248295744893994196900166571e-01 4.03974541969813680242218367766327e-02
            1.86849159371713937315140352524168e-03 4.43765239953598289446290436899289e-01 3.10025483435276659471835358772296e-02
            1.76531343516777954050223087278937e-03 2.29854890249822918502076163349557e-01 3.92854639000807109860957666569448e-02
            2.43665074530694967733235323237295e-03 2.51525427138907131663358995865565e-01 7.15641320496676891504961304235621e-02
            3.03484621096303339854438618772292e-03 2.20628791932752477134016544368933e-01 1.54394591780774664835362841586175e-01
            5.50881790375419870195439209226151e-04 4.16403182046817754713430304036592e-01 2.77347337277504936772776211739711e-03
            3.65459140776114863383150499487328e-03 2.78037243584027538023661918487051e-01 1.71845869035243042910465760542138e-01
            5.86809763869502467280303559249433e-04 3.42075832214508490647375538173947e-01 3.25241025804930876733078726203985e-03
            1.41387321020510284275140122645098e-03 3.76651522742024291723339501913870e-01 1.60811994772276506404651286175067e-02
            1.40590977903145685629082706213921e-03 1.12297916741235570547985389566747e-01 4.36591987048875543320214376308286e-02
            2.18881088644219939237656191721726e-03 1.86141155841773708878861270932248e-01 6.82956923610527899537459006751305e-02
            2.88989258748446366242834670856610e-03 3.21703398275187124610852151818108e-01 7.47225031108798287693417705668253e-02
            1.66084415751654133897030618527424e-03 1.66012685178866747914838697397499e-01 3.61553743643339420277271756276605e-02
            1.61710719288366602244474967697840e-04 1.93576952354284764767911042326887e-02 2.83309069924117659308726580036364e-03
            7.79979255879456709506236666840095e-04 7.19904092251254096312962360570964e-02 1.79498835307187742937884422644856e-02
            4.34229828650841581140495950563718e-04 1.33332716879897866801840677908331e-01 3.19253987057180636452424771221104e-03
            1.37583912829826286727818995814232e-03 2.99445529172993885058673413368524e-01 1.62924574190147324059818600971994e-02
            1.02529312215662404654537009918158e-03 1.17107885189153299032582822292170e-01 1.79153991813487192819032145507663e-02
            3.92075062823016567015860234235447e-04 8.53676016562654599306725344831648e-02 3.56724930413417637767303780549355e-03
            ];
        
        S2=[
            1.90023648880032858081434721242431e-03 4.72409482150556669921570573933423e-01
            2.40451787008430202877140580142168e-03 4.57507968673824261252036649239017e-01
            4.03949727267366963390227141417199e-03 3.53847291916883299744966961952741e-01
            4.19238340914585027631444891227375e-04 4.99106149860298997644747487356653e-01
            1.85446781068809006011444129313759e-03 1.13331658196365314683617953050998e-01
            3.34303455844673681491424632739040e-04 1.57787047228343108029591235208500e-02
            2.90176895980177960293611150177640e-03 1.64009184766605586869303579078405e-01
            3.84960331613813482032337986993298e-03 4.08471560729133986722416693737614e-01
            3.38825160904275340598146293302761e-03 4.36541655107956616532760563131887e-01
            7.96024816034627885272256797577484e-04 3.70326513393935824458580441387312e-02
            3.56278191707463378831710620886497e-03 2.11731749565185534134315048504504e-01
            4.36436425490537883986608136410723e-03 2.86957288459068593500234101156821e-01
            1.85384268403265112766520328335673e-03 8.06765828829424935797476337029366e-02
            8.74538907766669241887480534813903e-05 3.64321992072988865701832494892187e-03
            ];
        
        S3=[];
        
        
    case  40
        
        % ALG. DEG.:   40
        % PTS CARD.:  295
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.4e-15
        
        S1=[
            1.88776183926784619929520325776195e-04 1.67177838355710456363922844502667e-01 1.52794904901324161625164599342952e-03
            7.75842484927516764549315642085503e-04 3.71478099445237464060198817605851e-01 1.49718224721410841232538402323371e-02
            4.32207280373296347502493075509733e-04 1.59023884712373636540760912794212e-01 6.80425879912778096975101149723741e-03
            4.98592465936582786463493466300179e-04 4.62179329282665363187732054939261e-02 3.53139747015546109909500671619753e-02
            2.04538515649023167628298147668886e-04 1.14865167380080651637896949068818e-01 1.58398644760465636543989020879053e-03
            1.02183358004221368220409971172558e-03 3.34246008184714449296137672718032e-01 2.15166720844818110114360365514585e-02
            2.06322166313810970248582288277817e-04 7.23711649483900210100273397983983e-02 2.02526169256312012223464336102552e-03
            9.57348377198415415555154961424478e-04 2.23684246664788699554549111780943e-01 1.53612037490868675454480651865197e-02
            5.82001567994970244249164359473525e-04 1.11129016990584944535314093627676e-01 1.01659439479149828200466032512850e-02
            1.45933685532618497536494484023706e-03 2.76444541807790877818717945046956e-01 3.00138537838292723813893303486111e-02
            9.71169086161176634518221728598064e-04 1.69263355465675779942813505840604e-01 1.89700426400215629851864207466861e-02
            1.02185803211649072798350434965187e-03 1.19650955248758308901990687900252e-01 2.67407842221269662075222584007861e-02
            2.28089429217444387152161078269330e-04 2.83019097129268648593836132931756e-01 8.53818693537522336624301377838719e-04
            5.59076325349260700933728962525038e-04 7.04045501511467719524262065533549e-02 1.22584700439817353978533276404050e-02
            1.51823620580618487592439169731051e-03 2.16472098751412100892110856875661e-01 3.71700471134970150677290234852990e-02
            9.70092839555700982810682031498573e-04 7.69268815632590530206869061657926e-02 3.21942006634704513956535265606362e-02
            9.57461742196476315452935423877534e-04 4.29222732038551268995973941855482e-01 8.93215271131544107097166573794311e-03
            4.11606702811378490081128944666489e-03 3.25147924885204830935947484249482e-01 2.24045215227499733279259430673847e-01
            3.76976024293509265339352509727178e-03 2.78272710286936342249930476100417e-01 1.87626216990675664941790046214010e-01
            9.27587312436338961262050073486307e-04 2.85210253518710044051687191313249e-01 9.75852132386811378650381954003024e-03
            2.20241068708402912626476144453136e-03 2.72725631829831005603637095191516e-01 5.87062906216736027364966332697804e-02
            1.90360250309294589017572452149807e-03 4.15469594771239403296902992224204e-01 3.07419996861602630444654238317526e-02
            4.84722289527269656946723008772437e-04 2.20577446533709115028543124026328e-01 3.25675363711016695872357828989152e-03
            2.11204016674038661319223741941187e-03 3.42885620394358714424498657535878e-01 4.51876229604891294089341613471333e-02
            3.30966277647020811619738012154812e-03 2.31503860611680267744816319464007e-01 1.50818510224056762103472806302307e-01
            5.00703644413417830148771425768928e-04 3.81017129649182106909677258954616e-02 1.66736740374798728114846824155393e-02
            3.77815553726074332754891749175385e-03 3.52891108510167694767289958690526e-01 1.62105242145544004017665429273620e-01
            2.64935252199680537285586545692695e-03 4.00076069919581056577584377009771e-01 6.45258600999303599810019704818842e-02
            2.77734489806420736801584325803560e-03 1.87784341546537586475906778105127e-01 1.14379695468569117511314914281684e-01
            2.25299228047472037406162570505330e-03 2.07376937690341156539375333522912e-01 6.95719958527820786509110462247918e-02
            1.55656756946014333993499878516786e-03 1.07803830226375463308130520090344e-01 5.60343597005426088952795282693842e-02
            2.19491910810134171691543336635277e-03 1.47901466073126069966647833098250e-01 8.15896657841670980282344771694625e-02
            1.61927618604145763525625767442762e-03 1.60042668411358779412267949737725e-01 4.46292737712247408743415633125551e-02
            3.28714905580475931154427193803258e-03 3.77876669688772037769552980535082e-01 1.08791876599217379739670263916196e-01
            2.30968790187326963392680512221489e-04 3.92233696648084315117266385186667e-02 3.20614858452418710968445658693327e-03
            2.81016493113481072643744518302356e-03 3.24304445077138125963500669968198e-01 8.29765480589191445348618003663432e-02
            6.16384778464869049867391748165346e-04 3.50984847622504703146262272639433e-01 3.52073500998824817295740352562916e-03
            2.24505119098073043990004826753193e-04 4.20870827625693666451667240835377e-01 5.98294728636777515748401690132141e-04
            2.84409163267836670799248999230713e-03 2.52656038560264695203017026869929e-01 9.96837194601994935894140326126944e-02
            3.37395444806112181426738239053975e-03 3.01670333560527526017835953098256e-01 1.30823914375048672154022710856225e-01
            1.39280312524316573965371679477698e-04 1.60613879667907410997429451526841e-02 2.99558774001274126519422935643888e-03
            ];
        
        S2=[
            3.12803309837425254445048006246566e-04 1.57037453637930216054918730606005e-02
            4.29726447949339927095957492042544e-03 2.93889962897443068889913320163032e-01
            4.29966617078099584692285617393281e-03 3.70102868423741626635603552131215e-01
            4.01956779966071604814903395208603e-03 2.52696314852206671552892203180818e-01
            3.60591179745443045356889655295163e-03 2.10577129514804201049216203500691e-01
            6.14819425666627037774594755070723e-05 3.05751062854184036537974478164870e-03
            4.03340248372594835368776244877154e-03 4.03292222274262812664602506629308e-01
            2.33948811422645449459301048023008e-03 4.76751646800292860906012037958135e-01
            1.35864928596238546641694622252317e-03 6.48562451597025496630166685463337e-02
            3.11997101526516806488387167917153e-03 1.68896379782961736371049710214720e-01
            1.94545468980247544636685841368262e-03 9.62980341591576360915638588267029e-02
            1.51448455313863057127010680602552e-03 4.90701569130203962831870967420400e-01
            6.02514031127758531436455768925953e-04 4.98464579236597593592250632354990e-01
            3.61244286747944200882676213382183e-03 4.32271977606806356853041961585404e-01
            2.56819586307021523532978513060243e-03 1.30241606990209146665904427209171e-01
            3.04222693774763072033229249768738e-03 4.56871737154322776763848423797754e-01
            ];
        
        S3=[
            4.39274222643154955286748375442585e-03
            ];
        
        
    case  41
        
        % ALG. DEG.:   41
        % PTS CARD.:  309
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.5e-15
        
        S1=[
            4.73316998226450060156811350253747e-04 9.33847895869260646861675922991708e-01 2.67728022087155327624063971825308e-02
            5.38919164693542015499849728854542e-04 7.08749727266859141927568543906091e-02 2.14673861123130318984131292836537e-02
            2.76775703152319890298221904956222e-04 2.27676589099523561665616711024995e-02 1.19851112427548141953215221633400e-02
            1.87236045763998536564653996094876e-03 2.12702658519097947964482386851159e-01 9.64112026242689990063183813617798e-02
            2.03686960481118005047984631517011e-03 1.97079994941520175721905161481118e-01 1.27978222846432787562775956757832e-01
            8.94377285332314110426199516012957e-04 6.39130414360007115259776355742360e-02 4.23991066111561523688244790264434e-02
            5.15712369071735180631188555366862e-04 9.16254632757576847890135240959353e-02 1.16869974011746882713858042279753e-02
            1.26610363416353307805528594087718e-03 8.89175403557379839813989974572905e-02 6.32415329385831864739131447095133e-02
            1.12623927896231875583898940362815e-03 4.59230558903699015438348851603223e-01 1.48409448407795345470372438967388e-02
            1.40918754232970317753770306268279e-04 3.38843822348438572511497568484629e-02 1.84003961561307214853111524632823e-03
            3.05468653482411081276359787750607e-03 3.20075210406615351388381895958446e-01 2.58057113739871668567360529777943e-01
            1.65448560765117831661275005217249e-03 4.22068492531888939112150183063932e-01 3.62192254340186342576046740759921e-02
            3.75170187050299431315986131707518e-04 4.78764410729283981793535929227801e-02 1.05439660443664238315131242984535e-02
            3.11182961660478934681761664649002e-03 3.61799022706076223165894134581322e-01 1.51026887499565554140446010933374e-01
            2.45210435507164882171227127116708e-04 6.84569430922209731793515175013454e-02 2.48206090173236360030273317534011e-03
            2.94532213285211884670755644322071e-03 2.36288472596456022856159506773110e-01 1.54957767021962872222573537328572e-01
            3.11906005955700418169618259867093e-03 2.97895418985049564852829462324735e-01 1.51961553665098864085436503046367e-01
            2.94459094064066873117774258972190e-04 1.12557251523921778013637151616422e-01 2.37754573037215425485024766771858e-03
            1.19525780902384563537770023611273e-04 1.20658629713076021722217845422165e-02 3.04923808583817523149006589733290e-03
            1.16906687993456061790131794708714e-03 3.90653097086316902775138260039967e-01 1.50754657505242934539824162243349e-02
            5.22716485387803844395171459780158e-04 4.27410727373970178444295697772759e-01 2.88455410756848591377621282560995e-03
            2.99509456015478042756328136420052e-03 4.11845522097719740628463114262559e-01 1.04765137725845669525170933411573e-01
            1.14412479363062676365747627471592e-03 3.22221194516843667532413064691355e-01 1.48317675755721811031717649598249e-02
            1.80524352827792270402562646580691e-03 3.54524901630881295044872558719362e-01 3.67451935646271599100742832888500e-02
            3.52932890427508447889204390435225e-03 3.32027617298779720300672124722041e-01 2.02832977056254276648772361113515e-01
            2.18927628685137943762017975757317e-03 2.53029739084772276047630157336243e-01 6.55893176896967106115354795292660e-02
            5.08569179475154412896353850470632e-04 3.57039904894509707578009738426772e-01 2.83340788181253838001749656427819e-03
            1.72774226236023097265370740416301e-03 2.87060689211655395425282222277019e-01 3.60525157666744186513163583640562e-02
            8.31309620788597087789884021447051e-04 1.36131351631136071622663052949065e-01 1.41561971271275018058322103797764e-02
            1.67558532663435004461671695707992e-03 1.32224125661775021578492328444554e-01 6.37728096721327936124978918996931e-02
            1.19328232526644587539743636028788e-03 1.07924219839552332222964992070047e-01 3.35534395458731288242049117798160e-02
            2.08768949658442223155097039466455e-03 1.52886902141386588072080598976754e-01 9.95790446035427906368298067718570e-02
            1.93556074427131950300984275514793e-03 1.88101771929606642608234778890619e-01 6.46491520250393747470951666400651e-02
            1.09616959802194807033215440128515e-03 2.54982429150887923707813342844020e-01 1.48987268708079909768082771392983e-02
            4.56366168260760617959859164471936e-04 2.23078080481277696289765799519955e-01 2.85444727801385713522597242786105e-03
            4.01417326307675591257062563954605e-04 1.64050736021032539380115622407175e-01 2.83077182118177030772931956903449e-03
            4.89679800789312273667786001851709e-04 2.88151117418988578222638352599461e-01 2.83594679658896297452796986249268e-03
            2.97754338044403958282724254047480e-03 3.40957788874061529238446155432030e-01 1.05632686278177900796038102271268e-01
            3.37625766935877123092746288079979e-03 2.68293448069283901880055509536760e-01 2.05408895782746736546897636799258e-01
            2.46633730412294718131738235911143e-03 3.19409003528324686449479941074969e-01 6.74786310679263284795226240930788e-02
            9.99758561822006782543459380008244e-04 1.92295465185952280107173351098027e-01 1.49331073629515708595105749623144e-02
            1.42908494641375699574226132426702e-03 1.60414053402923612035024802935368e-01 3.53852934054426707111140615324985e-02
            1.62869147472471118356929231651975e-03 2.21253498980421736685997302629403e-01 3.62519958338577280576586758797930e-02
            2.93614082918626463289402117595728e-03 2.71194224068813549699541454174323e-01 1.06599696973037380565152432154719e-01
            2.54753922375026324934066401795008e-03 3.92323411299770385962659702272504e-01 6.65697076536746201735894601370092e-02
            ];
        
        S2=[
            2.61046615627579913897909902331662e-05 1.81005509308153171577071560705008e-03
            3.22163483544135337971470178786149e-03 3.14607751383877753958984158089152e-01
            2.82310188625230868916804460866388e-03 3.71832457839435748248035906726727e-01
            1.48755341237118794348037731367640e-03 4.81848960175834395425198408702272e-01
            2.12247788855559824106045141434151e-03 1.42281998308420726262468747336243e-01
            2.51812585448819457531466348143567e-03 1.74633614866341574378694190272654e-01
            3.00505023858112186793678510809968e-03 4.25198078415139901764518981508445e-01
            3.03861449848868373641019857700485e-03 2.09977372337917744804514086354175e-01
            3.38357010421123840376855795852862e-03 2.62739973625342737584276164852781e-01
            5.11819985150217099910407991814054e-04 4.98603639231114337260208912994131e-01
            3.61505486647585106210756045186372e-03 3.99566525526970250936642514716368e-01
            2.56693516261922473342682593511199e-03 4.66761057785514965612350124501972e-01
            1.98478451967980728740115203834193e-03 1.03095485576030487795406997975078e-01
            ];
        
        S3=[];
        
        
    case  42
        
        % ALG. DEG.:   42
        % PTS CARD.:  324
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.4e-15
        
        S1=[
            2.66365847170266191110132858455017e-04 3.30553853684353138309148789630854e-01 1.57042560146290624892295717529578e-03
            2.19538513584911982364639149167829e-04 9.69953965146683194653576265409356e-01 9.61466912071009052742631695309683e-03
            5.18512523267713283312008165637508e-04 5.71983061939756801117518136834406e-02 2.19035586156293565762709363298200e-02
            2.79775975114021928282642681651282e-04 4.53270971637945607746189580211649e-01 1.49355129029098685801424384322900e-03
            1.26896025323238598181418979038426e-03 1.40732473906720334522191251380718e-01 5.54414996743645416277246340541751e-02
            2.02907889905697873603185321655928e-03 1.83788630301941036115920269367052e-01 1.18693569421395000995644863905909e-01
            1.13321287603745650299051639375847e-04 3.21121524868778346339937002085207e-02 1.39305721718615584114875360910446e-03
            3.40926671570748252623056506749322e-04 4.29640722006761482942849283972464e-02 1.02994751818414106286914133647770e-02
            9.35465343078011708159019699238002e-05 1.05248427300748837592836437693222e-02 2.58470999228818361717618934392249e-03
            8.43314834911865944880315915099800e-04 7.53790285629702960790154975256883e-02 3.94743127161576515260499320447707e-02
            9.51274898228949877983240401846388e-04 4.36333362089955989482348286401248e-01 1.22938385245525295258461895286928e-02
            3.86683512649890282278364317392061e-04 2.70490746514976954095743622019654e-01 2.58980924784568732402445512263967e-03
            1.58619394545011973413073391725447e-03 3.50813643381097384210676182192401e-01 3.75563888722968697297943663215847e-02
            1.08426235493087105149490056987815e-03 3.72765511974292140795483874171623e-01 1.70945467207073396875838255937197e-02
            2.03705873682035690924729820494576e-03 2.17107352263341546461106190690771e-01 8.57583162943158783253494448217680e-02
            1.68604450719807524676496779392210e-03 1.92756365009005836697042468586005e-01 5.79856449641143953388500165146979e-02
            1.66873270767465067820900515016547e-03 4.20685030686416272516225944855250e-01 3.38481728829903155686409377267410e-02
            3.72150230356113544301355222643224e-04 2.12485185437091050886238008388318e-01 2.52669138520864814539956810790500e-03
            8.74502828754642911809891181462717e-04 3.19257208729187336171406741414103e-01 1.06565421155276375542175770760878e-02
            9.51584197719013472581417456552799e-04 2.51638235512572583374435453151818e-01 1.34296732422585305549711520711753e-02
            1.97907224537829636371033537045605e-03 1.52211981456809364310345245030476e-01 8.83417782932650746952063514072506e-02
            1.08166500595875442365667584709854e-03 1.11142094794033280402700825106876e-01 3.29221182111608043352468655484699e-02
            3.13337069827091061618995482263017e-03 3.58150058851250285840706055751070e-01 1.94800760777327719441842646119767e-01
            1.21573547237640439133699032936420e-03 1.63799856956150069464683838305064e-01 3.14610159618917972768059598820400e-02
            1.41328904880122724688207291876552e-03 1.02112773336674317326000505090633e-01 6.73683101645258797240956027962966e-02
            3.23767811274164101983941144169421e-03 2.33555703019351579285967090982012e-01 1.86565244199972779526319754950237e-01
            2.12424122562709320055640560553911e-03 3.21416462831711324810157748288475e-01 6.38834834123953448914434716243704e-02
            1.45504295453112702379838694355385e-03 2.23336184895712552833657582596061e-01 3.25712217949498672453501058043912e-02
            8.77197165465703663188301586473017e-04 1.90588118946320272950600838157698e-01 1.32734827561577940074899473188452e-02
            2.66390718176628916655235279620229e-03 2.27355840245479501326997251453577e-01 1.36233775509128074432396715565119e-01
            3.55867757526690030398658359800379e-04 1.56767922787160035591469409155252e-01 2.61537753388922135366900789676947e-03
            7.84259809071718516984561553329058e-04 1.34033334986778468467250036155747e-01 1.34812112051568063625373028457943e-02
            3.45484185169266664119924747922141e-03 2.95721175745453590533173837684444e-01 1.91884104352206263932600904809078e-01
            3.03883444669207994099507930130244e-04 1.06719271813000365911605626934033e-01 2.54728163216983664179915436420742e-03
            2.92995302527590019414649447071497e-03 4.12566419483916535515533041689196e-01 1.01037294355759685315909734981688e-01
            3.88259302031484327885624274756537e-03 3.07851982992886075152227931539528e-01 2.49750741165288780987197014837875e-01
            6.55020416493056162159092092878154e-04 8.70205486996966370849548866317491e-02 1.37126252885580324908376681491973e-02
            5.28548624586135085655358434308937e-04 3.89428331681219708215735408884939e-01 3.34459543340264765787228640192552e-03
            2.02059882537530040996776037331983e-03 2.61293970314177004254929670423735e-01 5.56746304306052139376781440205377e-02
            2.62123945672969088562659356611562e-04 6.49755331098118837918065082703833e-02 2.80375859354026635977819026379620e-03
            2.60243400316005450700895806903645e-03 2.71950006098130969700576997638564e-01 9.63505000772398451092115578830999e-02
            1.51665566313493606863394003170242e-03 2.93090364272546199941871236660518e-01 2.91991066004359950403213019853865e-02
            2.43427618019893430775790754694299e-03 3.94664826587819206338281219359487e-01 6.35745506621691741644397666277655e-02
            3.30143630904448232010750530207588e-03 3.56852218192982795219592162538902e-01 1.43928580065543848531817161529034e-01
            2.82656834415620158984516763212014e-03 3.41361351638876264757982426090166e-01 9.92086997532997810766985935515549e-02
            3.09596450617196814955600636665167e-03 2.89512941079514585229759404683136e-01 1.40388921421462892213938289387443e-01
            ];
        
        S2=[
            1.77322000798858358389109568520325e-03 4.02371925109793249220047073322348e-01
            1.73597677182810482883913022478239e-05 1.36539257595404031446484349743287e-03
            5.07084821416367960769666645148845e-04 4.97531729625205365596940509931301e-01
            1.42209297850611968150325026272185e-03 4.79900295754419115024091979648801e-01
            9.35490769011439789876183326100545e-04 6.64663290557835906291472838347545e-02
            6.66312500203990274588727871218907e-04 4.32363893877459951053943143506331e-02
            1.09634084079294211525046343069789e-03 4.90305014201646682803215071544400e-01
            4.76202977827340642128645109565355e-04 2.69988226283729981780368234467460e-02
            2.17992502094623355479718895821861e-03 1.38504806910322769697430089763657e-01
            2.26626612751637535048132576775970e-03 4.67013207355763959593986101026530e-01
            1.84803533077693843411637431017880e-03 1.06580425858858796051009676375543e-01
            4.04866880451419490494213704323556e-03 3.11735391072587297855989163508639e-01
            3.64530052639338610076524815895027e-03 2.44860575868881713557101420519757e-01
            3.02776339101268549339196489711412e-03 1.74835145811190639797416679357411e-01
            3.37947231961488372270485669446316e-03 4.27569334150190383248002490290673e-01
            3.96102228829565625178865673206019e-03 3.74270156678447651898977710516192e-01
            ];
        
        S3=[];
        
        
    case  43
        
        % ALG. DEG.:   43
        % PTS CARD.:  339
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.5e-15
        
        S1=[
            1.95631917040878450589347359311887e-03 2.29739436527249235586722875268606e-01 1.45453795387832596208710356222582e-01
            1.32877988603856021440630108543246e-03 3.60558076088783818757832477785996e-01 3.24154748417785493197484925076424e-02
            5.20192928808181349713268826207013e-04 9.59172276669436746354691081251076e-02 1.24129082244383796146403042826023e-02
            1.93441112276208134618249445679794e-03 1.89801510111088478360841236280976e-01 1.37858117891493076045961174713739e-01
            2.71800396079252982312057795866167e-03 4.01058451292605144811176387520391e-01 1.35336131153596217480128416355001e-01
            2.56651053850672546813305174850939e-03 2.77582532276102134094486473259167e-01 1.43526791892476662804156717356818e-01
            1.36244263020324045398989909472220e-03 3.06118376690335036727219630847685e-01 3.54515915965141689691897397551656e-02
            1.54096391385130150088189626700341e-03 4.18289170173424762477765170842758e-01 3.54185331313221704596649885843362e-02
            1.09800733044825536699473644830505e-03 9.65099243357523811637221911041706e-02 5.57938681257981461403261391751585e-02
            1.35358581459473436246765398749403e-03 1.32839741110529208079071850079345e-01 6.39762347898086281006158060336020e-02
            8.86625283151499360943803562662424e-04 9.69643297980663232404907603267930e-02 3.02625797417711110692817300105162e-02
            1.05970476604279979435108138829946e-03 4.28753753691012928772607892824453e-01 1.46376163589685189675426713051820e-02
            6.91132913699826095278722082326794e-04 1.36853445326083039690345799499482e-01 1.36052509293985018912209739028185e-02
            4.14389801137990507369501802514833e-04 3.63561909099252844601579681693693e-01 2.47604062219227219476813672827120e-03
            2.88046498518415615766596893365659e-03 3.36143572327619710726764878927497e-01 1.42925973377929421692655864717381e-01
            2.43320326789467198834102101834276e-04 9.40988534499003353994694975881430e-02 2.42077774579726101231158530424636e-03
            3.03058530224832763100284438451126e-03 2.42710586669900529876997552491957e-01 1.95150339989513393890163683863648e-01
            1.97958017650065945525206245747540e-03 4.25238442831057605886257988458965e-01 6.40551018216570233754225682787364e-02
            9.75765280742350881761082437293453e-04 3.64278126974187321795994876083569e-01 1.31876112324832255040529815914851e-02
            1.58793200978925015502685180734943e-03 1.77663726489636492056334304834309e-01 6.18730821070126515981435488811258e-02
            2.08884489201240173411600054009796e-04 5.88456479492050557600357763021748e-02 2.61247234276882700845789919696927e-03
            1.40189925650187514893918461211797e-03 2.50099601909829272194940585904988e-01 3.41904035600949715578700249807298e-02
            1.53716150062059976697664831490897e-04 3.19725865365346195723361688578734e-02 2.65083133054744551210246683581317e-03
            3.79023937257839135892695248131190e-04 2.41723961357679040240498125058366e-01 2.59156908972399966165078843971514e-03
            8.14280093595709107863467313137562e-04 1.84404707439201814844764726331050e-01 1.42733464909807362258620244688245e-02
            5.00790757565847707670492194864664e-04 6.12563890152719517367962964726757e-02 1.38564076854345488015729515041130e-02
            9.48517546105485172703436491659090e-05 1.32571416922718544884052249699380e-02 2.49382626599786065904607035292884e-03
            1.96129685941321546538418374439061e-03 1.37973140081455225569584399636369e-01 1.00440931058371943351836819147138e-01
            3.67632224535556658123053308528938e-04 3.32056799273112990267264876820263e-02 1.36974423527236825043340573415662e-02
            1.31387504913329394330701038740017e-03 1.94466196781106465119037807198765e-01 3.41431958895440298973333881349390e-02
            9.20816959513956327133821222474808e-04 2.39482635085204537128689139535709e-01 1.38046252946412654344365833480879e-02
            2.96251638633688012952616119832783e-03 3.64554227776340777911201485039783e-01 1.86497214299555097438698680889502e-01
            3.06169420407145147309396771007073e-04 1.37356402366199809028657341514190e-01 2.54175936468376819851733294797214e-03
            1.91248172423942900885474038830125e-03 2.32232037318890388588599194008566e-01 6.30349337046809815587877778852999e-02
            2.06415161110312835035518475024219e-03 2.92661395072246866000398313190090e-01 6.38769125793660075274260634614620e-02
            3.34601602087061496607067745401309e-03 3.17648116398156710182831830024952e-01 2.41952845175475161898503984048148e-01
            1.02364147369649028350424568856170e-03 3.00301596615778321197609557202668e-01 1.45992712589763495162342721300774e-02
            4.39007839582545406845875968215864e-04 3.00561353716810975011952677959925e-01 2.81811459692608715435491184564398e-03
            7.88892559272938963255139555030837e-04 6.05239793987312202361295021546539e-02 3.41344995593797426169935249617993e-02
            3.15371795549420318172106902920859e-03 3.00785919233889365020928607918904e-01 1.93355477276418330490059815929271e-01
            2.13631991052658362292837601614792e-03 1.89550703977685197632041536053293e-01 9.73629697501508795109970151315792e-02
            4.84918673119629948468217728319019e-04 4.30139903362148912080442642036360e-01 2.78830240511142692441670654091013e-03
            3.69009135228931463464097095794614e-04 1.87018572217664685863880436045292e-01 2.75596244648371960922483836498031e-03
            2.63094068006375922733064420810933e-03 3.79189522509007448380913274377235e-01 9.78271071588677626484908955717401e-02
            2.65013598749912595090028055722087e-03 3.12966976982159250653126036922913e-01 9.95736856368709677411743541597389e-02
            1.22790050513958476016695975374660e-03 1.41804487017288982642071459849831e-01 3.41143516662648707549188031862286e-02
            2.52780519757884233195133205640559e-03 2.48181460015588495338079155771993e-01 1.00497443272928421720280312001705e-01
            2.21709548456510557867815691679425e-03 3.58797835407998699785991902899696e-01 6.16738036965516905896222965566267e-02
            ];
        
        S2=[
            1.53998278459081643088079971448678e-03 4.11780170859933791760454369068611e-01
            1.24367411012324185701216627819576e-03 4.75575867099131466275707680324558e-01
            3.04307861090390044216114073805102e-04 4.99179836734477966331269271904603e-01
            2.93904116687957774056716431232417e-03 2.96247988168869824665563328380813e-01
            3.43424448325747736290858291852146e-03 3.53188804887000662446894239110406e-01
            8.07153621957876926153174768074905e-04 4.95105980802769907178628727706382e-01
            5.41811214635524204634442479999734e-04 3.29098159605345705336354455994297e-02
            1.35830253414758860691646802365540e-03 4.86563751130688337820373590147938e-01
            1.01216465227482689190341424989583e-03 6.27150483897616628503612901113229e-02
            2.12271596053216190028156695746020e-03 1.44743559770221352778207801748067e-01
            2.36051052789603479096017246696704e-04 1.33646206082615737198437955157715e-02
            4.28662883399411019104928433076651e-05 2.56494137002013589468618626199259e-03
            2.66763767119809454622658684286307e-03 1.87772242778578402067068964242935e-01
            3.47484043452828952544542140401518e-03 3.82564308786935480810598164680414e-01
            2.66640271020050648523302960768433e-03 4.52892260845951022396604912501061e-01
            1.64535307292699821413162197814017e-03 9.28670462140478714108127178405994e-02
            3.65710457156016984989199691824524e-03 2.52484932649090720868656489983550e-01
            ];
        
        S3=[];
        
        
    case  44
        
        % ALG. DEG.:   44
        % PTS CARD.:  354
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.4e-15
        
        S1=[
            2.05987823218479039564382526705799e-04 1.97551676157197361738404595143948e-02 1.07567195566842938037055432687339e-02
            1.36620423001841104259160286638064e-04 9.95165813818446182148491629959608e-02 7.80585899861933429737248513191616e-04
            2.69664221964948671271533608972959e-04 1.53494544755213874198673806859006e-01 1.93833291279059036736620580398949e-03
            3.35489485291019459440448047260475e-04 4.60607452003998885103897009685170e-01 1.78614248052541058943420981819372e-03
            1.12965768056970034308693351388087e-04 2.96420604264443915409898977486591e-02 1.75422517105275517501628002037251e-03
            1.61033154709168884822015344049362e-04 3.33840624824811804671753634465858e-01 5.17294259660069986710151379583067e-04
            4.83883651397566802096483762340995e-04 1.19426687227411967850976282079500e-01 7.54587092570611038533368031266946e-03
            6.73332446090880933249611661750578e-04 1.73311806699428916367367037310032e-01 1.14925464528052804352720173142188e-02
            6.68754141990617967730714532592629e-04 5.93978633857566162834729084352148e-02 2.42629677211045335916139720211504e-02
            6.04714018344116059286430520103295e-04 2.59446164627843145478180986174266e-01 9.79769330245964208125908356805667e-03
            2.98688070880941548401049656646933e-04 2.69664879531117784861038444432779e-01 1.89915082820833660409709509053755e-03
            7.34100531866785648189843715982761e-04 3.23428410410956435061535785280284e-01 7.67467985614434661245653046535153e-03
            4.00527242600620767432745283898043e-04 2.09397609435192477711495939729502e-01 2.70720430776859224270003423384878e-03
            9.27835211688663732357978952691724e-05 1.07937373867395408622016361732676e-02 2.70639655716989335765276791789802e-03
            3.60711013213061608764786747372000e-04 4.16577600266789249694454611017136e-02 1.04627340296394866214768626377918e-02
            7.66083362023714802176155291846271e-04 1.39300774418482237182459471114271e-01 2.15128602849198879987113741663052e-02
            7.54513310214147614116142470663817e-04 9.71407574555290531659323960411712e-02 2.28245356349519729066699369468552e-02
            1.70086123647450893764387913620340e-03 1.51124809738970578010253120737616e-01 1.00088530304790021552285850248154e-01
            5.17994081903193023287690355260793e-04 3.93932395535154467047789239586564e-01 3.15956953663550782485103241015167e-03
            1.18874384638993136655005233848215e-03 1.27637459417324766741685948545637e-01 4.35675752808351648970308644948091e-02
            1.69769561709233940058338951928363e-03 2.15342785917052303990004702427541e-01 6.28153762458395631407270798263198e-02
            1.35204419124091042292423736626006e-03 1.80523887256028525705175979965134e-01 3.46911866169311577934308843396138e-02
            1.00850169514564479492091386703123e-03 3.70697478749728304148192137290607e-01 1.59521357680311760829372502712431e-02
            1.39594056621288563346805489118196e-03 2.38904346634593867104712217042106e-01 3.65656909809639746322673659051361e-02
            2.15022943286524888251154408180810e-03 4.03544504764916933670093612818164e-01 1.22867248406971205088034082564263e-01
            2.49192826459688088897026503332199e-03 2.51088618401061669516138863400556e-01 1.22061192033510804155760354205995e-01
            2.32664813783349565604052777700872e-03 3.10092460353001919681759090963169e-01 8.78278870328067923223258617326792e-02
            2.06393802112825276370844385098735e-03 4.17188937068111109773838052205974e-01 6.56811030561964964702426073017705e-02
            1.02891088117624640400904478809707e-03 8.42771975158688041451782169133367e-02 4.57201471848305632250664132243401e-02
            1.94691022921440661658243342913011e-03 2.87517281890495191021273058140650e-01 5.28034108950670294513329849905858e-02
            1.38538069827777205085539424800345e-03 3.42142184906992441462136866903165e-01 3.32627075502785821137230470867507e-02
            1.24007199925338222092074946800722e-03 2.90781335627484260886177480642800e-01 2.30909511650298354679122780908074e-02
            4.31451090416774016116085999072993e-04 7.77385297161002769561477521165216e-02 8.41212288864582984071471827292044e-03
            1.60466588787147622970769011985226e-04 5.83538087981567143991412649484118e-02 1.85584469043908713639656760108210e-03
            2.49050430530872030088218593846250e-03 3.53736217391187435232779989746632e-01 1.51861221499153403735604683788551e-01
            2.50978895904076454784203953352062e-03 1.94103683421922135199366721280967e-01 1.46693846495503765758883218950359e-01
            1.38461640914676505087488322942590e-03 1.15244680262074267185212761432922e-01 7.30582724234847963318273400545877e-02
            1.85079032983941570676689192254116e-03 2.52648918856827386925090195290977e-01 8.33205145800466895344982276583323e-02
            2.53751872958100054000762924033552e-03 3.70638127911035064609990286044194e-01 9.76700995193537646610337787933531e-02
            2.13383834577409239874334900832764e-03 1.97510766235328455442399331332126e-01 1.02127473235820234731541233941243e-01
            8.64613651994068188426367793653071e-04 2.19102178776639594515174280786596e-01 1.69401226467935060726599516556234e-02
            1.61236995612197607855020642375621e-03 1.64749471455534146624444247208885e-01 6.46074146543942490206191564539040e-02
            2.64397385648095124832668290082438e-03 3.48756887430614948009122144867433e-01 1.94293491861925404773359105092823e-01
            2.00979214318253726970220363057251e-03 3.55853330058662320478646279298118e-01 5.87895877753212212080136112035689e-02
            3.03041660365002685217916678084293e-03 3.48716528985469398360663717539865e-01 2.43584975885173138365757949941326e-01
            2.89956852928164405744371201478771e-03 2.90704335549696046836487539621885e-01 1.86125965956163291759040134820680e-01
            3.19657376055502377015793413761457e-03 2.89862934817113671037702715693740e-01 2.38357019552556226305028985734680e-01
            2.82733750449421861530852062571739e-03 3.09383575445457970065632480327622e-01 1.31229616791972980571756579593057e-01
            2.65552470844998772872558312485580e-03 2.46979539099244443933756087972142e-01 1.68458978552479210488712624282925e-01
            9.86615210737950458844136569780403e-04 4.35345195121465189380671745311702e-01 1.31254695112234757298219633980807e-02
            1.77773973578347144544609115257572e-03 4.13701289314883091652319535569404e-01 3.44134405185802239035552929635742e-02
            ];
        
        S2=[
            2.21520454570737544241869343419893e-05 1.71230757996369982974760315386220e-03
            5.61488691411858175002447257639915e-04 2.91673514653004867236418817810772e-02
            8.72575782766473384208238961434745e-04 4.95499879123024178362300062872237e-02
            1.36482635824496686786921895162550e-03 4.87723980583153360335302295425208e-01
            6.02253006705962250916241984555199e-04 4.96061804465474931458146556906286e-01
            1.18289665538962485320795270382632e-03 7.65961156051737368510856640568818e-02
            1.75559713417060736691255318220328e-03 4.75951846052771243567036663080216e-01
            1.47173402359788724896860756530259e-03 1.09095541957237654462353759754478e-01
            3.36162144923654766792697756727648e-03 2.31247003693840935678593950797222e-01
            2.48992940625946106852905970185930e-03 1.95908198223650492630909525360039e-01
            2.46041862262338451375343062466072e-03 4.22321808207151372371868092159275e-01
            2.51073751429880089383606112107827e-03 4.54881749959505921054869759245776e-01
            2.22329662038065660814578627935134e-03 1.43900119540420812214165380282793e-01
            2.32330638229220247728545878374007e-03 4.02340447854735383526048053681734e-01
            3.21273602508716444106640075517589e-03 2.94295765690833932559655750083039e-01
            3.13279394181341592010014984737154e-03 3.51615788795768935415253508836031e-01
            ];
        
        S3=[];
        
        
    case  45
        
        % ALG. DEG.:   45
        % PTS CARD.:  370
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.5e-15
        
        S1=[
            9.71532123877306627579564235475118e-05 1.26048038307361616461310660497475e-02 7.29242955084338090321338654575811e-03
            2.96832401489166389262682604899268e-05 4.60547663144127931295068734129927e-03 1.34999718211049760455133572634168e-03
            2.40344353751331804222579613394828e-04 6.79464966039111523121718505535682e-02 6.85917593806053670774103281360112e-03
            1.15062466750112129165901919680692e-04 6.93380043634408277375413831578044e-02 1.22976799593489888665076570362089e-03
            6.47231358534072396225533818636677e-04 4.68270859870947964154197507014032e-01 8.78010020069475149995241736178286e-03
            2.30260449145343131481522558345887e-04 2.50495170335221506441314431867795e-01 1.68302636118027710253330919698556e-03
            7.75345040014331513076456525723756e-05 1.79231013398374397493917342671921e-02 1.75797258353514685921825666525820e-03
            2.63216008449837707890844873404035e-04 3.59456841995052789329179177002516e-01 1.47561365316294709065791934676781e-03
            2.03880140043860092272098860632923e-03 2.21153984784947493791307238097943e-01 1.53663127462592530303453486340004e-01
            1.36366857778165971072648066098054e-03 2.38973194878696115761584906067583e-01 2.27625216137249258840924426294805e-01
            1.90891918002091031184019875865943e-03 2.26522885981175758907113504392328e-01 1.14203037751511859987374464253662e-01
            1.62179496424432996806719309290656e-03 2.39590524037790186806518022422097e-01 8.03750594685513908288498896581586e-02
            9.42482859192454399711869061917469e-04 1.78291402992592273868410757131642e-01 3.06183636824900254347348038663768e-02
            1.09036119380349146197606291508464e-03 2.82586689120417511311700309306616e-01 2.69421027814230988617349993319294e-02
            9.17290619580522705207858980003266e-04 1.37257110258729553287082580936840e-01 3.27972135469326506185261393966357e-02
            8.13669897716355183109326887347379e-04 1.00254415236569610625494419764436e-01 3.27271127336489636716621021150786e-02
            4.14266410586362537806642825088943e-04 5.17074765599652777514627643995482e-02 1.51601013138725621010038935310149e-02
            2.55374504338384605492406898008539e-03 2.60614855137656920014421757514356e-01 1.70884652952977555973035350689315e-01
            1.11955667944577791882154738090094e-03 2.27214324959957303384783244837308e-01 2.86632862041672348274445880633721e-02
            8.20086182210520036640899466107157e-04 3.11040487235050899794686074528727e-01 1.26956715659609486318570858998100e-02
            1.83583569658887020732696004898798e-03 1.85623876299053219840473616386589e-01 8.79015352488810713937894547598262e-02
            7.06502470799716180901994810881206e-04 6.58626437608177683902255239445367e-02 3.07203461767424183814778615442265e-02
            3.82221000431039507026775847720046e-04 3.01590470095362273017514098683023e-01 2.81455063987638438166571930310056e-03
            1.97701581302660257463688076029484e-03 2.92542657654236315512008559380774e-01 7.98394163722915967129623027176422e-02
            3.22416772478558073290744179217882e-04 2.73024847528849125277705667258488e-02 1.33211979027671369862906303183081e-02
            7.77378451269039290674744968612231e-04 3.71571920577544234642886067376821e-01 9.60218601811555806657683120874935e-03
            3.13239896240613069702818371808917e-04 2.01007303993687103105258984214743e-01 2.50243738523270699627976121348638e-03
            2.99590287508693341178567237648167e-03 2.93183129372266426670989858394023e-01 2.01730828065996542486359999202250e-01
            1.52920890728200733511832343225478e-03 2.33331891103622568461872788248002e-01 5.26776823006486205613896345312241e-02
            1.49820395207656519021011121139964e-03 1.79299141897004327139697466009238e-01 5.57691461107685942932832290352962e-02
            1.61668679165861538943316233485348e-03 2.93435331129092979995931500525330e-01 4.82367013092171934007801326060871e-02
            3.32318218248998887773271171397482e-03 3.29840596275619235555609520815779e-01 2.33697583553122173460891985996568e-01
            1.59745246645443244454254383413172e-03 1.38175693611527594661936291231541e-01 9.26621506087761254155665824328025e-02
            3.08099584148887337478606163188033e-03 3.58512865776101086190408295806265e-01 1.73454768103072393126140582353401e-01
            1.09190886564487625066954823438437e-03 4.20337783466548764454984166150098e-01 1.78508936465828042805537023696161e-02
            2.41911437679862874328140520674424e-03 2.79051133003480611449731441098265e-01 1.19946692495847162529543084019679e-01
            5.85471759754056385489928704402018e-04 9.34626722493710621053608633701515e-02 1.39846172517608963864477189531499e-02
            2.70374842821992106972617264659675e-03 3.87408987689019879141483215789776e-01 1.19357764315077496952710589539493e-01
            2.26669456187451271583777057117004e-03 4.07184695661520157372592620959040e-01 7.35846549853617637459990419301903e-02
            7.63422222268319563762140056439875e-04 1.87640583389709708628245721229177e-01 1.27167478485407797750283975801722e-02
            2.99277879376055888378110880765348e-04 1.52225526038276298690377075217839e-01 2.44038040768785029904774930287203e-03
            2.11066871570363308319162598536423e-03 1.74142001786563360710147208010312e-01 1.29890625977294477122470084395900e-01
            2.75728111566975056870298921296580e-03 3.21408155175526122349083379958756e-01 1.43880630387304531137004914853605e-01
            1.16858890495533432850316923889977e-03 8.88043361734264791174098263581982e-02 5.88598092883567541866440819831041e-02
            1.39399447633985098908560029684622e-03 1.30428915551712210074697395612020e-01 6.02973413848093622857682305493654e-02
            7.15287497685285125065268729827039e-04 1.36837912722210020000090935354820e-01 1.32254525993588060300831799054322e-02
            1.98602112093872488693402367232466e-03 3.52853359179670589274024905535043e-01 5.72712595019886847036438837221795e-02
            8.11510714695544812039007531723200e-04 2.46386900217142157387684164859820e-01 1.09336475862860226654360573661506e-02
            2.41542743894754751918241275632226e-03 3.43520044059471285624596248453599e-01 9.59251650895475566072079232071701e-02
            1.76983940807325340442701655518931e-03 4.16208963636464202018316882458748e-01 3.96704297157567664644517435590387e-02
            1.83532994709030859497786436485001e-04 3.90118106999857716665758289309451e-02 2.79504775052941568241893222079852e-03
            2.85484954702091219772702723744828e-04 1.07180714134606072662769804537675e-01 2.66661165267688318042393902373988e-03
            4.38056769584411532549028667204993e-04 4.26423819706326090450687615884817e-01 2.53953007625778645617309337012557e-03
            1.54355548746208420980707831660084e-03 3.51960131363829642481277915067039e-01 2.87313755582887971062877596750695e-02
            ];
        
        S2=[
            1.60120857517628855694324929714867e-03 1.29806653300276658713485744556237e-01
            2.31867442056098941455255513233169e-03 2.06349665630883027933606399528799e-01
            2.04353023274365074024583321943282e-03 1.75084588806939833727227551207761e-01
            2.41556761516013173861014418086768e-04 4.99504576867959404484764718290535e-01
            8.46305070277620198962442721324351e-04 5.52204190605027900318191313999705e-02
            3.41687768784906592198225006029588e-03 2.99152765836593892423422857973492e-01
            3.44625239929348100090256146188494e-03 3.66982277256458211400769187093829e-01
            3.15732849651645299665636557051585e-03 2.64921642390380518428116829454666e-01
            3.27915375009894642654995067232448e-03 3.98469068413945026740918819996295e-01
            2.50842974298410865269248759545917e-03 4.52620190751494178016400837805122e-01
            2.94125456308765367841040472285385e-03 4.26861049533307446157692766064429e-01
            1.56857173108213639352304102914104e-03 9.45724371334943636480119266707334e-02
            2.04653257390983740054979378442113e-03 4.72989470868897310396050670533441e-01
            6.25657342660230217715189304783507e-04 3.40851411567991424700352354193456e-02
            1.52790288514941668962521248431585e-03 4.86998996825781460540127909553121e-01
            ];
        
        S3=[
            3.47073482719251679914918007341385e-03
            ];
        
        
    case  46
        
        % ALG. DEG.:   46
        % PTS CARD.:  385
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 2.4e-15
        
        S1=[
            2.22685409346649702821974159405727e-03 3.97969764640709822156594555053744e-01 1.54123149166077250127315778627235e-01
            2.16000476232081420335795485243580e-03 4.14307607988175785607154466561042e-01 1.13136705924023417413337710968335e-01
            1.54343982796946075329580505552940e-03 3.29728249876623691339005972622545e-01 7.30183277797032820322087331987859e-02
            1.71511290774016209037533542414167e-03 2.25759242330057757763839276776707e-01 1.09803376807363559097829863731022e-01
            1.89138099119562754293899509150378e-03 3.28699347560523658096087729063584e-01 1.02835875371072957285001336913410e-01
            3.35215320031303522490401247679870e-04 5.70245183292808974218068840400520e-02 1.04550153444100853150633412269599e-02
            2.08705738367026419691208083406764e-03 2.18153490627607082874561683638603e-01 1.47522261546056410175253859051736e-01
            1.27918332848800907208808297355063e-03 2.02306345883463578605443444757839e-01 5.21260968074300848340563163674233e-02
            1.59199347606164116029903166094073e-03 2.13283209808807994178891931369435e-01 8.00841957661688891301210446727055e-02
            6.58585463976954320472131931829551e-05 8.21335887122196829646103566346937e-03 2.16078035466830617461475583240826e-03
            8.22836191553000750008306507510270e-04 4.61515361612300334392955392104341e-01 1.13638023497272810935498910112074e-02
            2.29774154662927511047487705297954e-03 3.54622467312705691444563171899063e-01 1.29778523617651048160936966269219e-01
            1.43443007840780931833046807355458e-03 3.10920643586031608673891923899646e-01 4.89372786072454005190657255752740e-02
            2.36336703046011132808845722763635e-03 3.04955784133230656252067092282232e-01 1.58417559007465957865434802442905e-01
            2.40436588441475529442925562761957e-03 2.50794437563904126786695769624203e-01 1.75500085438409114679814138071379e-01
            1.59638354184672356939067583425640e-03 4.33869221864480514305739689007169e-01 5.31263813104962887012838734790421e-02
            8.74382525144714976839122044793839e-04 7.70376986253196172071966429939494e-02 5.09089773583295684966643079860660e-02
            7.15335166191503803861939658759184e-04 2.76836186019522423329419780202443e-01 1.11213995971377495319831751885431e-02
            6.34087639467665101743187872074259e-04 1.73637162103421421877058605787170e-01 1.17797536807179720197424899197358e-02
            7.89516871955837917784748469784972e-04 3.35433688661376960826743243160308e-01 1.18081263036933694493546553871965e-02
            1.60772260421602661804862943739636e-03 3.75064218452396624936540092676296e-01 5.12438839908936349942791821376886e-02
            2.69885291100847727122080677020222e-03 3.46711286129900431163264329370577e-01 1.82541256476132840447235139436089e-01
            1.53135260150543935434019893193636e-03 2.55968511260664111084395244688494e-01 5.39018714502928694698447031896649e-02
            1.15960920481872494479380630849619e-03 2.92006920788531632027229534287471e-01 2.73433134615536990730255695325468e-02
            1.14805014508864147368405461691054e-03 2.36417239579575555463719638282782e-01 2.97423818792802684141118874094900e-02
            3.66905166035599872038225344184070e-04 4.66150898129435442562851221737219e-01 2.19129468105736533237171137500354e-03
            7.21880138370402748594634179113427e-04 2.23482629145585642849169971668744e-01 1.25736658366042697299036490221624e-02
            8.29333161982129989445167161932204e-04 3.97368013878000581051708195445826e-01 1.15196691295125427934031847598817e-02
            2.78536465405662290659161151040735e-03 2.85679243244989944372491663671099e-01 2.10668037666766339022217380261282e-01
            3.60718717232299049268995361217094e-04 4.00991912296406793903003062951029e-01 2.17085028246803788537500068400732e-03
            1.31106391220738678279889022348925e-03 1.56273510404076748647739236730558e-01 5.57664223797979460850449129338813e-02
            1.51789801686191111983192936207843e-04 4.77858936846818488364263544099231e-02 2.09790827030163079536251302670280e-03
            1.05004350057420303525546056988560e-03 1.82249159931374082876942566144862e-01 2.90781975795925198835512048844976e-02
            4.83149309370665416222406784996224e-04 8.93475681919859154866969674912980e-02 1.15829668440252560313030727456862e-02
            1.60233618511448601753888176801865e-03 1.65219881239912769199662534447270e-01 8.74623616502727824206786522154289e-02
            1.96147522693670739489871235328167e-04 8.05280047391684350444052142847795e-02 2.07850359821926643222322894644094e-03
            3.12879060401823972173418475506423e-03 3.30541501567599038668987532219035e-01 2.36477783744783759400931444361049e-01
            1.89062025704545846015169896503494e-03 1.75074013252293603226661389271612e-01 1.22809018053712887263984043784149e-01
            1.15318098238607956267098852976005e-03 1.13394973029422363453022626345046e-01 5.50050284444838430419899566459208e-02
            7.98215168649468861591933510624131e-04 9.45442284064488130956860345577297e-02 2.87935332271121169345207135847886e-02
            6.24418006796969627704430028103388e-04 5.89589423608462975812116724227963e-02 2.62933317918677932067961222628583e-02
            1.27781420414726587866938167792341e-03 3.55446117102369674167761104399688e-01 2.83073273852235927916520807912093e-02
            1.33914246584527444478007751627047e-03 4.20627844445428888953131263406249e-01 2.81269388071388806715766861543671e-02
            3.20695007916892962842453362881656e-04 2.76342477166445721348253528049099e-01 2.10090109306926720728059265752563e-03
            9.55358677029266643680738280153264e-04 1.35834405138137270796505617909133e-01 3.05221004022598642324570761275027e-02
            1.51428242832448768130648719676401e-03 1.21432444920387294651220599916996e-01 8.81430419815303345387746958294883e-02
            5.97009737256883055156464479296119e-04 1.29254481332948201188770553926588e-01 1.25808072722038699620128454625956e-02
            1.99489307211004945549759348466523e-03 2.72939318330326308181810190944816e-01 8.38760556578321686549415403533203e-02
            2.38707993822376155282349863284708e-03 2.76224431674286952098640313124633e-01 1.24791600444704789407701639447623e-01
            2.84050648765330404285633925098864e-04 1.66531061991801065946816606810899e-01 2.23024928943949115808798033810945e-03
            3.64644894945766217305566092576896e-04 3.37323861878955577608962812519167e-01 2.28466711100102455242466525930922e-03
            2.19977808499302032382849425573568e-03 3.89984837680378781943346666594152e-01 8.31933025800658670911857939245238e-02
            3.39479499288666082459225981082795e-04 2.19057402029606912430281795423070e-01 2.40703109183256977465781289993174e-03
            2.65722056703089572118198224970342e-04 1.20341342443090318359288914962235e-01 2.41706361847834373668408858293333e-03
            3.23762331706669464256170831717441e-04 3.21434597854267645189452196063939e-02 1.28868718184530495207695821591187e-02
            1.30663973001150102029935262670790e-04 2.39783625881974243876371133410430e-02 2.58739871201071761422363159965698e-03
            ];
        
        S2=[
            7.53181193949790809064048779930545e-04 4.79162709481021131807665369706228e-01
            6.23734590822670764177770852398908e-04 4.86830795833199034272631422481936e-02
            1.09475819165005274456736739097096e-03 4.86849434452578500032160491173272e-01
            2.26630825917091289350979721461954e-03 1.96294809229449501719599879834277e-01
            2.05606677560804572760244646190131e-03 1.65798716567095938412279565454810e-01
            2.66814199893165805363426379415159e-03 2.29007117787044606194157836398517e-01
            3.23781096391130606207409137198283e-03 2.99898101879728595786644973486545e-01
            3.26415847271570563198350711786588e-03 3.66081261411425384899587243126007e-01
            4.85854850068560926035915459308967e-04 3.07693439520817652033102262976172e-02
            1.21934895073770761808606977183445e-03 8.24986937635582839867254278942710e-02
            2.09410203588819305436929951724778e-03 4.60823004347588105478195075193071e-01
            2.99313714823754711824577157130989e-03 2.66615708697696740525628911200329e-01
            1.88674007826653087477319914455620e-03 1.28439434947512254314716528824647e-01
            3.24082362724326632219540300638982e-03 3.97511575402048289440415373974247e-01
            2.24673355140021959027926690133370e-04 1.28796984358850331170431857685799e-02
            7.04761006797523675046057006765565e-06 4.92922242405799231057539344647012e-04
            ];
        
        S3=[
            3.31752356160357718406483940043472e-03
            ];
        
        
    case  47
        
        % ALG. DEG.:   47
        % PTS CARD.:  399
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.6e-15
        
        S1=[
            1.06391003262472504825066066391770e-04 1.32182347742614610813793873944633e-02 6.92610811899467895824678009830677e-03
            1.12574137246430547310005465533322e-03 1.82074143261559212580991129470931e-01 8.16358471109855377001451870455639e-02
            1.13253505774401643478588308511235e-03 1.50580927317701368517433024862839e-01 8.61963659887330785336345684299886e-02
            7.01364787340122354704352591880934e-04 7.64329826567074904763998688395077e-02 5.02931805301914183758960064096755e-02
            4.89049584063298958922123838011231e-05 1.38326160976293081189547606868473e-02 1.04889894991745235425240068138919e-03
            2.01931493401746409242086821578255e-04 2.67452272430761042731894860935427e-02 1.00204975653522664752470561211339e-02
            2.57682766030653672903460993381941e-03 3.71190236752514501894495424494380e-01 2.03674253045179914911599894367100e-01
            2.25473552819196619220742938694002e-03 3.68894972792118358562163393798983e-01 1.59326683873933672153100360446842e-01
            9.57294483603011070634025703185443e-04 1.07074579893119942175516712268291e-01 5.33687025646959883884434816536668e-02
            1.30313259149965050658070975231340e-03 1.15217912167360214681366414879449e-01 8.52960644923164285069461243438127e-02
            1.53807863714332249127636487173731e-03 4.17962853969899517103669950301992e-01 5.14822979290668059082491936351289e-02
            5.16406156255945101782312001148512e-04 4.62992599159318674040264340874273e-02 2.75536314394735394606161804631483e-02
            2.17718061520642217759879954996904e-03 3.25029937315999439828573258637334e-01 1.18702027168940849044531660183566e-01
            1.69604395891152000958035372946142e-03 2.23021456853453786672858427664323e-01 8.50615771696930367262723393650958e-02
            2.26657835230176241125832348188851e-03 3.82240080251783131970455542614218e-01 1.18127508441192408117359491370735e-01
            2.68485965378233680522779280863688e-03 3.18681016804057470892530545825139e-01 2.52708350652741664532641152618453e-01
            2.50562594513383408295625898176695e-03 3.16102429197291978635320219837013e-01 2.05745941208171606584542701057217e-01
            9.76697137394437302215788321291257e-04 2.01714360262501668508150487468811e-01 2.86457371054149485811723963024633e-02
            1.13507664579970499879602030546266e-03 3.04095374457296840908782087353757e-01 2.87008250916218343729013895426760e-02
            1.06933649152424333980682114031424e-03 2.50940600094452903601194293514709e-01 2.90822358487935334059315550803149e-02
            2.13906709163620614017053966904314e-03 2.70113685487746946201070841198089e-01 1.21742235940660512016897598641663e-01
            1.25350967222140298115706791293178e-03 4.23007886771842922524200503175962e-01 2.79469184369985149118953415836586e-02
            1.98686555019924509027373638048175e-03 2.18838108408261983184317500672478e-01 1.22140066758372467514703885171912e-01
            2.37020684316891069517985179970765e-03 3.15558282430156267217569165950408e-01 1.60412985091024695716299675041228e-01
            1.58045107889223895684949194873070e-03 3.59994205076786699670776670245687e-01 5.20384051030055791442485713105270e-02
            6.53246515891859735186697477615780e-04 7.61492805627096702725253862809041e-02 2.80080973264347880824498560059510e-02
            1.21120828529703975819098449306921e-03 3.61687125042856494339105211111018e-01 2.83641992932530291604553696060975e-02
            2.06687929519422343080381487823161e-03 3.96957042979479091115990740945563e-01 8.18149550216045046147073094289226e-02
            2.55256659060561306301417872077764e-03 2.63582104650385729005535040414543e-01 2.12133896316561210149131966318237e-01
            1.34881166699170763206838685732691e-03 1.92535054212271156126590199164639e-01 5.24452042229162709863210523053567e-02
            9.40482198953648991512910271239889e-04 1.55053798001932918726453181079705e-01 2.91776764327326132431039695802610e-02
            4.57109505137131048001231103938835e-04 8.08139183670610050658211775953532e-02 1.16752954894914842687292377831909e-02
            3.50532801434540500998965351087122e-04 4.94903005695827707954848051485897e-02 1.13529551848956314352712126947154e-02
            1.59505282435752539921258730259979e-03 3.01366119175060842572122510318877e-01 5.27412086548005240582703834206768e-02
            1.21691292055276672513797198149632e-03 1.45559094124306370732213622432027e-01 5.39363844971338171618846502042288e-02
            1.25423725728734752025983478773696e-04 3.39239311459584025332247847472900e-02 2.02689658380067255424861194512687e-03
            8.55406563880252565276740561728275e-04 4.60920398298352129717869729574886e-01 1.13741139787508632741364067442191e-02
            2.26400606671470043529392768277830e-03 2.09650636992742611663942398081417e-01 1.64719735798059069287191391595115e-01
            2.00406961507452151260988415515385e-03 3.35618830727636319188889046927216e-01 8.24852827778116137746877711833804e-02
            5.59770199953664644200446165456242e-04 1.19198356094802629390194681491266e-01 1.18660258094383762317303165900739e-02
            8.27824540613834630904677158724780e-04 1.12751742717588893105684633155761e-01 2.89885801232179704522984309278399e-02
            1.53727860563678746068372316102568e-03 2.44604484993710485918327890431101e-01 5.36092519706683826030690909192344e-02
            2.41503521209857335144688406103342e-03 2.61327895213015415354362858124659e-01 1.64967292586388120101403842454602e-01
            1.94934870191071189793330376005542e-03 2.76710222358847013346405674383277e-01 8.41334586353880142794992025301326e-02
            8.22925995577372020386985873585672e-04 3.31057698305513958825230247384752e-01 1.17181407852650371764768877369534e-02
            1.85595243546373637327784322792468e-03 1.71165960353522994541464186113444e-01 1.19825159275385270674085802511399e-01
            7.09287803930804374233365372504068e-04 2.14838093540415636661933262985258e-01 1.18084740509011697912056249037960e-02
            6.44529097785457162454736401713262e-04 1.64032595788237256506647554488154e-01 1.18982357159540252933282999947551e-02
            7.80745027452893263983368932201756e-04 2.70599520519274439056545134008047e-01 1.18476002079960143070769262862996e-02
            8.47635945473001984826200150280329e-04 3.94881529905421246873231666540960e-01 1.15294467270000370456850546929672e-02
            2.26401136283958608584609972780299e-04 9.76801440609630089451087542329333e-02 2.24970582857548151087256904645528e-03
            3.72669915533183288350316253456640e-04 4.31663188476496095802303898381069e-01 2.17808311913864462508461095069379e-03
            2.66174073629335745157925563475487e-04 1.40323860572580283445631721406244e-01 2.25943109747260215125486126908072e-03
            3.28809841813197927667955511310538e-04 2.43796900107501685051047957131232e-01 2.24895104491058166543138163717686e-03
            1.81034525565338457213024336844853e-04 6.19939934379152152166270184352470e-02 2.21078294406797885157667238331669e-03
            3.53005972835246427474903407883744e-04 3.02950492841990437486998644089908e-01 2.24993142954393399515766738261391e-03
            3.00938551109985498455134944606471e-04 1.89266100502910289415936517798400e-01 2.26403926204082287659824501702133e-03
            3.66319673274832519754840021164455e-04 3.65919027936809082568458961759461e-01 2.21351192100762176814821025061519e-03
            ];
        
        S2=[
            2.15732847916675924554441579061859e-03 3.49177504154390971891075423627626e-01
            2.34958561562595208069281760288050e-03 3.02896544117914334748320470680483e-01
            6.18246204852672832236182287601878e-04 5.05319024623013812735905503359390e-02
            2.25799868909655043064388380003038e-03 4.21509392878709365781730866729049e-01
            1.47224053313712893893694388935955e-03 4.74627024342958780955825659475522e-01
            1.04150124319736482256248955025058e-03 7.95728420692438376260469112821738e-02
            3.31503931633386999915102943958800e-04 2.36802291430402538086052999233289e-02
            2.79912721630153722360279111569525e-03 2.63056435770372187832322197209578e-01
            2.27590725678494375933125226652010e-03 4.42069397263830543565177322307136e-01
            1.25364067529043217257000986109006e-03 4.86160107483689762908340981084621e-01
            2.03221496293360098100988508917908e-03 1.62146122522450308878205760265701e-01
            2.75255677067570659818929179607494e-03 3.73691181805182015551736185443588e-01
            2.47233079993487399675378135555093e-03 2.12551268904813739357706481314381e-01
            2.03289490379707964498590122559563e-03 4.59832174149501116122706889655092e-01
            3.74415430601989678623814361557720e-04 4.98918529148673406403702301759040e-01
            1.81606992783187688386692393294197e-03 1.24974506748911293185777537928516e-01
            4.46268307090295687626115006541028e-05 2.60509669473600602440654938618536e-03
            ];
        
        S3=[];
        
        
    case  48
        
        % ALG. DEG.:   48
        % PTS CARD.:  423
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.5e-15
        
        S1=[
            1.85996570614835451604185401208724e-03 3.69567291590166213666179828578606e-01 2.15700644169616900791552893679182e-01
            1.51630547228510242918630979147565e-03 3.08481897993362230714353700022912e-01 1.08301499922003982567808577641699e-01
            9.23505432416745126490698059029683e-04 3.19642729224551336297821535481489e-01 3.46959527711364415503147995423205e-02
            1.18864540424328977638590565391041e-05 9.96255965477151894660323705465999e-01 1.34606393094949803625193229805745e-03
            7.72483726320483598745680886565879e-04 3.28084817808831308294514883527881e-01 1.91707840803216067460112981279963e-02
            2.51594349134969669234962452719628e-03 3.19812908272719464974898073705845e-01 2.76361565310809742523900922606117e-01
            1.94367883400491786843183872335317e-04 1.58926601361820019064907683059573e-01 1.64116834381746313879790744039155e-03
            1.14648466765973297033787137166883e-03 1.96275977197235607096814646865823e-01 5.33231934213235156394539160373824e-02
            1.62950409788547418386828180558723e-03 2.93342184590181742631642691776506e-01 8.29288364011383782870012737475918e-02
            6.49717196700759264289815053672328e-04 1.24727325881397566043062852259027e-01 2.33190171491815048665507958958187e-02
            2.02211589957075480242232323746521e-03 3.11593245943279584242446844655205e-01 1.38713093959385580511423086136347e-01
            4.32239553547259793778861913793321e-04 1.10541441031880752210447838024265e-01 1.05406394081225263104073519571102e-02
            6.32985718237871167364538838029375e-04 2.97417196275917228298624195304001e-01 1.03472505675079231440838256617099e-02
            1.37129429762075628028428209148615e-03 2.94068058652278530029633429876412e-01 5.37818816587767034587130865475046e-02
            1.05770377683287266669387438611238e-03 1.11401217606310831698124275135342e-01 6.97603580204464307490397345645761e-02
            8.11577263642842541488398033777685e-04 4.43743453863629289113390541388071e-01 1.55023027792510516287283195424607e-02
            1.27063686670264748557190870315026e-03 3.67635711929402475561801111325622e-01 3.96225072588169413556613562832354e-02
            3.11317736056086497714590066721030e-04 4.83834188936242623468864110236609e-02 1.18107009236640553284658494703763e-02
            1.32092018998813628440081036075071e-03 4.26198546034848302355158011778258e-01 3.52878923463722377062978807771287e-02
            1.63265855137869416065232730517209e-03 3.46403707508342595389194684685208e-01 6.47150930019751435828823105111951e-02
            3.82757292961209119510485043846870e-04 7.61686440792014585188240971547202e-02 1.04225715676421054611822114566166e-02
            2.58206804723576290733211280326032e-03 3.19855805989206987138828708339133e-01 2.27104917530668970471197098959237e-01
            9.27719490637918123634997336068864e-04 1.14976749976327827229560796240548e-01 4.31548880805705872831801173106214e-02
            1.13990500568115615126485451469307e-03 1.55341187486334963985257218155311e-01 4.89749482183904280563346844701300e-02
            5.13397308598045193361869564796507e-04 1.53664139478065231436687554378295e-01 9.45454884202667593384727950933666e-03
            1.66421560563311248404777531462173e-04 7.98538207718234027687742582202191e-02 1.89430587911264334925742769399903e-03
            6.07874655725859028981838960703499e-04 1.97860112128548371046576903609093e-01 1.20283224036929169142862150465589e-02
            2.94655786447838821846822998651305e-04 3.16915380380781819269486732082441e-01 1.90446777793631645894767423499161e-03
            8.67132902247430689332552411485722e-04 1.64563977163413477367015502750291e-01 2.62052237923564648736096671655105e-02
            2.02493753788023089779102292595780e-03 3.61129835979937918732929347243044e-01 9.44709215240264049695539938511502e-02
            2.13711614885642295011414226735269e-03 4.19825977454229393170948014812893e-01 9.63233490337837422456956915084447e-02
            1.41675427379012262157909063375882e-03 2.43175224574663090271187115831708e-01 5.34903172634274073904414592561807e-02
            1.37190627294187135476466465888734e-03 1.45983223113643451052112709476205e-01 7.95244786008371812302542025463481e-02
            2.36510728177526086644932856728474e-04 2.72392832679802729400897476352839e-02 1.10307547814719290879548196926407e-02
            6.86658872080503010493224280708091e-04 2.45313708964778415255736376821005e-01 1.21862329639788515706078086964226e-02
            2.11650495730306950669177834534196e-04 1.16554219073452897115217297141498e-01 2.11321826247760956654642150454038e-03
            5.33230593581922210182078014639728e-04 5.35856828700446083391106810722704e-02 2.72189906580712175088176962844955e-02
            6.91200536655851697533103905612961e-04 8.51355181500168534114436624804512e-02 2.62562590124720725281992628197258e-02
            1.82925589651349658655565821874234e-03 4.08492625785633056700874021771597e-01 6.24750974503704217810629018003965e-02
            2.60500242743329050998379869952259e-03 3.82251045094813779456899283104576e-01 1.73776209876193493686358237937384e-01
            1.03655280295382467865145104113367e-04 2.59740322065890247293928894123383e-02 2.04912604306957185293369683165565e-03
            9.79158143431578618237631772558416e-04 3.85689188760064305405705908924574e-01 1.88900416615420362054234004745012e-02
            1.60446126495937969186345029015683e-03 1.90746479431403104243258894712199e-01 8.26189529456984456112778048009204e-02
            1.57207577306592505761093092608860e-04 4.93146516048518085417740053344460e-02 2.27995678675310979241497300051833e-03
            5.39257277560204980638991489882983e-04 4.26482545498046594634189432326821e-01 4.77225212030264397955470201395656e-03
            1.80021348830760653282612171466326e-03 1.51364752670565094749832724119187e-01 1.16202356411588061946282834924205e-01
            1.83059570892130747329917817012301e-03 2.41085560390986441969829456866137e-01 8.55252642423670605964503010909539e-02
            1.08085615693754048469155115697049e-03 2.12036211205479691033204403538548e-01 2.97767883228066514156306965332988e-02
            9.00528138193089182361728006753765e-04 7.73420280842538471199887339935231e-02 5.02535370358713970317943164900498e-02
            2.72377964713793954687481857490639e-03 2.65927023654531302643277967945323e-01 2.16300207102354397070698155403079e-01
            6.98926007621861877924449024668263e-05 1.02781124040352035697853949614000e-02 2.31820768447709487508623382723272e-03
            5.98100592597650593425229192945380e-04 3.63748680489815068295200717329863e-01 6.43018778976174251726938280171453e-03
            2.47628291179179072997729171845549e-03 3.69820205106713650877736654365435e-01 1.33111115065471463658752782066585e-01
            2.52760293694821117777005170523807e-03 2.62236855625541442904591349360999e-01 1.68252485371562954252766530771623e-01
            3.02795530949602985891500050996683e-04 2.05865274198998832178020279570774e-01 2.37764370084020187565143622521191e-03
            2.67889510546185944522834532222078e-03 3.21061299239171094033196141026565e-01 1.79275176914734907240855932286649e-01
            1.18276761225013398760863569947333e-03 2.67008925352811121722851339654881e-01 2.84633516120191137899197997285228e-02
            3.18088027822487865326622769046594e-04 2.58604541010351018925916832813527e-01 2.22646709189586958915763048594272e-03
            2.05838971587918176733844610737378e-03 1.99624762677360428897443966889114e-01 1.20364369572249552131637528873398e-01
            2.37178308508622726416548154304564e-03 2.08890864384406460541043770717806e-01 1.64334461686715643802259023686929e-01
            2.30113371257409209114097592419057e-03 2.53333600601808928143299226576346e-01 1.23887458837676739387134716707806e-01
            1.17504461240287695167165693543865e-04 4.63796122746566341632501462299842e-01 1.60010389384323214144456581919940e-04
            1.19936839465554427423821248765989e-04 3.83717799050546426453678350299015e-01 1.08509201743662683412559866180658e-04
            ];
        
        S2=[
            2.18256937689136933100630955095767e-03 3.47724458710809425365795277684811e-01
            1.98458752265129193328996315415225e-03 3.73431271255322050350855533906724e-01
            9.11057268392255308053773177334733e-04 4.89339053562853232293150540499482e-01
            6.05723625032489809347224785085473e-04 4.84109248251398810447554410529847e-02
            9.65477244813535070977472685171961e-04 8.01595487479755675064296838172595e-02
            1.32001218782053686591482222212335e-03 4.79090265959213501556490655275411e-01
            1.64209604320303490659904710469164e-03 4.66642415348434647093256444350118e-01
            1.61472479217407850326376772542858e-04 1.17045744671133034736643097062370e-02
            2.71552315199755393490543831092054e-03 2.67846513711168798810291491463431e-01
            2.57628027224190633837075381507020e-03 2.13138272480190765767460447932535e-01
            2.49650257497091620020879254582269e-03 4.33626661366610932635268227386405e-01
            2.12174924592708081388936491862296e-03 1.59900914440538832739235886037932e-01
            4.43800280512309886036398376418788e-04 2.80636634926781647347837633788004e-02
            1.58696417448596360788570791555685e-03 1.08047129242305917085076316652703e-01
            6.47539955694782353584626477527308e-04 4.97175637238466083811516682544607e-01
            ];
        
        S3=[];
        
        
    case  49
        
        % ALG. DEG.:   49
        % PTS CARD.:  435
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.2e-15
        
        S1=[
            7.58974691375411498681091693185863e-04 1.74709854307173090282034877418482e-01 4.00081550569362995317668207917450e-02
            4.82018154443511221557416845939770e-04 3.86965057163826176012122459724196e-01 8.66241723145184591703404919371678e-03
            1.40675183266099958776310097086082e-04 1.26060933435405392311068339949998e-01 1.30110914835385301988046613530514e-03
            2.34311066873256089101995591761352e-03 2.83924952000500496751556056551635e-01 2.45628505378345851628552054535248e-01
            2.25894949919161047019455779150121e-03 3.30248269334462274393615643930389e-01 2.50741636173609283577690121092019e-01
            2.13206740649609559562840943591766e-04 2.56045941068157811848493565776153e-01 1.50973749388043430690042878694612e-03
            1.62341831505261047219335068803048e-03 2.60395316134476195735203418735182e-01 1.08983380212925196062201393942814e-01
            1.35567494687072317635645468669736e-03 1.70712696379342027119818681057950e-01 1.01968716908776124707713961470290e-01
            8.16379113471230131310329980465212e-04 3.74644058095124909968376414326485e-01 1.97722885748598982413870572827364e-02
            2.39515088089517871102612778244634e-04 4.35168457293983068634446453870623e-01 1.35418528809368014266534263612130e-03
            1.30013591471941632009279299353466e-03 1.39092391229103939931377453831374e-01 8.76428525159974741720603219619079e-02
            1.53690383100287594059230045218101e-03 2.80150654301227930442763636165182e-01 7.84155789651995310807208738879126e-02
            1.49390992511364874284485182442950e-03 3.46331187427559528124731969001004e-01 6.27477726380917394388703201002500e-02
            3.46295674839279337374475797872719e-04 1.14244793358781793202183507673908e-01 7.10573045383911233024987197381961e-03
            1.13209474794829438419607736676653e-03 2.60147782714403918991052933051833e-01 4.08904742282701419431312217511731e-02
            1.08418096863569630126933329705707e-03 3.38595450369722106298553399028606e-01 3.24948830275603520179039662707510e-02
            6.30167241237812006567076128504823e-04 4.40524947160692070813325926792459e-01 8.72904428993618967946854780848298e-03
            1.50964002721019170064759018501377e-04 8.57234837407636202089022958716669e-02 1.57475866401919737010139943578224e-03
            2.28081151372815831360196803245799e-03 2.85984086639778534166822510087513e-01 2.01532205535708414112505693083222e-01
            1.32157138195153627154010411715035e-03 3.02243064852707965517453203574405e-01 5.14181635143590096137522493791039e-02
            5.65729964641722353306529935679237e-04 1.14557425728885831084369328891626e-01 1.90287488235584209550488310469518e-02
            1.05443659419094011575612768183419e-03 2.12296430019530718569242822013621e-01 4.07880592078401632560868961263623e-02
            7.80409350560663751458778669700678e-04 1.46528429709901986122133621393004e-01 2.85146118965765860142624887885177e-02
            2.31542555620518116649142470464540e-03 3.74081646108826992858098492433783e-01 1.53708658611640291979938410804607e-01
            5.82418912261698385607333605662461e-04 8.17068303997550704487196071568178e-02 2.46053203840888995435065567107813e-02
            9.53923793556404149215910415193775e-04 2.94147478880844692739060519670602e-01 2.35355019299825565792794179742486e-02
            1.81036796651772285277059992836257e-03 1.75070369482651533310502145468490e-01 1.37716721260120800174320265796268e-01
            3.02352958305667494139784556850259e-04 4.76205378057730083218856975690869e-02 1.09753100892318270626057952199517e-02
            2.23578901441128657934243051386147e-03 3.17572481513326898561189182146336e-01 1.60465657202724848451325101450493e-01
            2.16089327797228244673610753068260e-03 2.88138053995077414448644503863761e-01 1.32949267737699799996065053164784e-01
            5.91722984001433215985044089535450e-04 2.67970889303296155858902238833252e-01 8.75495123131142180539221442359121e-03
            8.54923059629563819421249615260194e-04 2.39379059203026878233089291825308e-01 2.07437619833565727034407899509461e-02
            7.63033388765984666238240130553550e-04 1.91853271569703082155555762255972e-01 1.97602896840749810147830345385955e-02
            1.37645103473179545444660121233937e-03 3.87324242534184959296794659167062e-01 4.29392857768670643814346021827077e-02
            2.18161361839562398592939906905031e-03 3.53532415828410828684269517907524e-01 1.17892418158290449015090928241989e-01
            4.83142678116276716882709907352478e-04 5.21774890148561099056934153850307e-02 2.53747294625837269133050710934185e-02
            1.89077633403432346717587542173078e-03 3.23702116151313945557888018811354e-01 9.35293820848460272232571810491208e-02
            3.01784656373465436183384236201732e-04 3.72059360674848627059674299744074e-01 1.98710348807312189509111099994243e-03
            2.12758198041948206127860210301606e-03 4.13692534607104922539377866996801e-01 1.08690196293505916824706503120979e-01
            2.42452666120458043552687854571559e-03 2.50865443641437146915507128142053e-01 1.74902334150963623127239543464384e-01
            1.51274590766758087155563483605647e-03 2.36847703538783593835148622019915e-01 6.69659133716484916698519214151020e-02
            9.15132955474600405521012902454459e-04 1.09957683042648218796344394831976e-01 4.09791337707322012029464985971572e-02
            1.15564557263624252680944692883713e-03 4.29731902895181538948321531279362e-01 2.44664150758457620005170696231289e-02
            1.14880906414655372185795734196745e-03 1.43768037433372491973315732138872e-01 5.66880883410084066009737568947457e-02
            3.86871300727660279248493191417424e-04 7.74106779986907661550432635522156e-02 9.65248336154636912831428219305963e-03
            5.54555395331635353310673064441971e-04 1.55706005665346897126610770101252e-01 1.10995676273211840018317886347177e-02
            2.95864003539375022486246358255357e-04 3.12853302449651204941716287066811e-01 1.97407178793647343148509598620421e-03
            2.33164539169912796807634935980502e-04 2.60902661381798156581268699483189e-02 1.10588672805732885634499851335022e-02
            2.53027257107086504372278623620218e-04 1.65672860817991568493923182359140e-01 2.46254442236619333975178314233290e-03
            7.73102621795123757034162714774084e-04 7.42257546763113590371219174812722e-02 4.63950129826373944275097471745539e-02
            1.11908386263531291833668579460692e-04 2.88339172370140305046160023039192e-02 2.07859992487419847906338077336841e-03
            1.47809259194698673113566034231781e-03 1.86822660672315066232584968020092e-01 6.83532967983220413055178710237669e-02
            7.14520924585108517176901266054756e-04 3.27083928185687322631736151379300e-01 1.06763043029804560041240435452892e-02
            1.07293467420879360882390862741431e-03 1.03891103433826778590365336185641e-01 6.91692834636133124792678472658736e-02
            1.79238370972849715007102489039426e-03 2.15534288242954619185809406189946e-01 1.00668809434643721778535052635561e-01
            2.72636788911361600470928401307447e-03 3.43151613027389945642653401591815e-01 2.01217937528444273942440645441820e-01
            2.19083288782858124840724478588072e-03 2.22408758723325439277118675818201e-01 1.43453248492636881161743644952367e-01
            1.53587267982228020462773243615118e-04 5.34809769143942265179347828052414e-02 2.12516053769432446476916886979325e-03
            1.61908850822945903254190813669311e-03 4.38788968845026938758735468582017e-01 5.14225024494020452325848680175113e-02
            2.05269142946377632433230253639067e-03 3.91932101375555841382691824037465e-01 7.81526329255782525873286203932366e-02
            5.34179366171052541142905134563534e-04 2.11082179897034566407043598701421e-01 6.49692444592179802614895223200620e-03
            7.11599458074916257042941136745640e-05 1.17098573109517579970972889213954e-02 2.09030641915285923110312538142352e-03
            8.02792352761153981278302582680340e-05 2.06383353759989840447985898208572e-01 1.39863518147532501665827875392267e-04
            ];
        
        S2=[
            1.61493883722624406891477111258837e-03 1.78373016229631531359700602479279e-01
            7.89545893009191012031078305000165e-04 7.28406536060113435349094856974261e-02
            1.96545273941011984855653516035545e-03 4.28988355639407015207353879304719e-01
            2.49792961248308320945654692479820e-03 3.81168837845570263045402725765598e-01
            2.03895945143986543485192797220407e-03 2.00738367597118549934265274714562e-01
            5.85893274137246197405537451174951e-04 4.62782231758690903999919896705251e-02
            2.54202687899602415219635354048933e-03 2.97143088863247917252863317116862e-01
            1.45496719895626676170852231706476e-04 1.08421217690070140204916526727175e-02
            1.23399131936961325810286904669510e-03 1.02853381663129031520220735274052e-01
            1.22377744593757820795731916518889e-03 4.84173563416694874295131967301131e-01
            8.41735588392922848624544407414305e-04 4.93261195566052690342218056684942e-01
            2.68271127030140317179518483214906e-03 4.05732471975522424045834668504540e-01
            1.74963401588083366892945047510466e-03 1.30421644925131446912658361725335e-01
            1.98000989575437781262112757474370e-03 4.61434805630296263068146345176501e-01
            4.15975444451907908852833495316759e-04 2.75739602431112201808005579550809e-02
            2.68220453544746250881813942612553e-03 2.31842445375473937829724491166417e-01
            3.06338680377703101789799156051686e-03 3.51721776627771431922298006611527e-01
            3.23057149542252543912221329769352e-05 2.21242130998175424647089748475537e-03
            3.94277582726634801511822470132529e-04 4.98789564691512810057361093640793e-01
            ];
        
        S3=[];
        
        
    case  50
        
        % ALG. DEG.:   50
        % PTS CARD.:  453
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.4e-15
        
        S1=[
            1.73664079746238607208919102475875e-04 4.43323086858342041427505364481476e-01 9.22456106070470324524712690816841e-04
            1.50025050954953880367331864320590e-04 2.59368938757903871872301237999636e-02 2.24379912623130783888569794726209e-02
            1.36520102961489335144373846731014e-03 2.40144105263269458383490473352140e-01 2.22790728293732892550949031829077e-01
            5.66485725892312548361562374310552e-04 1.34552618026642700765904692161712e-01 2.31202139702463074777405438453570e-02
            6.82609204983525633125829301661724e-05 4.50355724673314194994588888221188e-02 8.56779795008972286721182776147998e-04
            3.84688033392161323788249749355828e-04 4.08714566567771886429483174651978e-01 4.80863766087200571186510700272265e-03
            6.48486597306067980538146766633645e-04 1.71463457739983227146041144806077e-01 2.34930123594385593943645318404378e-02
            1.40292758105521587173869502152002e-03 2.10105924705033142441124027754995e-01 9.33966471293635747130679192196112e-02
            8.54412111496703828399101876556188e-04 1.39296820118585046310855091178382e-01 4.17966366511481543510875269475946e-02
            7.41763471609338921955445478317870e-04 7.03820539127592015304912820283789e-02 4.93573118928612927791554909617844e-02
            2.12709721662030035865820210005950e-03 3.72606488156493997898621728381841e-01 1.52814107012161137788552878191695e-01
            1.09889979335918972803975979246616e-03 1.47915717277644076510512149980059e-01 6.51154299306215683662557580646535e-02
            7.74290973945819028557979724070037e-04 2.14024643581643692069604867356247e-01 2.34621431760208033756587298057639e-02
            1.30164026507678491639818929570538e-03 3.30267403058144004557306061542477e-01 5.20322782344110751351351495941344e-02
            2.33085132051627237203400566656342e-04 3.69984863544403907997804026308586e-01 1.49451910553232629290387745157886e-03
            3.28788578186756336261403754761545e-04 6.75115690433361337330353535435279e-02 1.20437469347925395213305677089011e-02
            1.39323270474968623384992572056262e-03 4.37098841180096731040549684621510e-01 5.66847855434612496461355135579652e-02
            2.06482847657084300774243956766441e-03 4.14512221157785298863984735362465e-01 1.14860142044815705109073178391554e-01
            1.07714145866373242101110552226828e-03 3.62380512268927423846776036953088e-01 3.12648361867088886434373762313044e-02
            1.73797610083554479251666258221576e-03 3.12251285511203513500788631063187e-01 1.02246640145541758948333210810233e-01
            1.35330712719854769250649173528700e-03 3.92689321703687865827703262766590e-01 4.83493018730729307019799989575404e-02
            1.92889643491063456342410331600945e-04 2.60638541622880276493745554944326e-02 9.97957692964265841162330161751015e-03
            1.71983321461280976721885949132229e-03 3.98390995844955408955456732655875e-01 8.39129407699283902388742717448622e-02
            2.19013468545185504589289937626972e-04 4.58803913209149261120955998194404e-02 6.72813170128427201488463538225915e-03
            2.20300726390513460645204091292726e-03 2.51098307865166847641091862897156e-01 1.77308024832368460455001013542642e-01
            1.00875040422768896299088314805203e-03 1.83136933934611223495636522784480e-01 4.26840877350856129357481449915213e-02
            1.61913930227482747228884285561890e-03 3.54046303278362339828078120262944e-01 7.43927680400381513559082691244839e-02
            2.33310539959698639081442372855690e-03 2.99210987110167137004168580460828e-01 2.43568947797036305935591826710152e-01
            1.57961290887743640611040607524274e-03 2.90839232985411644261120045484859e-01 7.63915697459878195729743310948834e-02
            1.73679225922312389235324658898207e-03 1.86146131853306973091832787758904e-01 1.24224589668886137294734339775459e-01
            4.47696440204899093976559365515300e-04 1.33627174971224615962483994735521e-01 9.77900837056553014214888008837079e-03
            5.40239241812688771558759182056519e-04 6.89763351482116093826846281444887e-02 2.79432020481641396614236327877734e-02
            9.06487224554971822258753189061053e-04 2.61491019269808000480281862110132e-01 2.48082685108521189265573525517539e-02
            3.82578565243165921207213475696562e-04 4.39794560591167671415036011239863e-02 2.11260004278838765812142952427166e-02
            1.42884085213951955196931553615514e-03 2.46957231707310764790008761337958e-01 6.81498702540452999665276934138092e-02
            1.32790604527584558090802513419248e-03 1.91607302979841132017924110186868e-01 6.87953069004346856596399106820172e-02
            2.23871881055322848735888996429821e-03 3.17395982308531277027441319660284e-01 1.56353477120021455704090840299614e-01
            8.26758630636716812523090425202099e-04 3.93839306299842706327041241820552e-01 1.57887711356747735547134681155512e-02
            9.72494817857270368519073677759934e-04 3.18144668212785697303246479350491e-01 2.34464079672303087054086034868305e-02
            2.02725879035876854761255572157097e-03 2.80364487557996810984661806287477e-01 1.35929813471386190526501991371333e-01
            1.17359850493396799045453526844085e-03 2.29975730084156670951500700539327e-01 4.46955970130420640606949689299654e-02
            2.70989834094576475739812870813239e-03 3.49488125037135055617198986510630e-01 2.46056250689159622391244397476839e-01
            4.10709557778086238512965655544917e-04 9.64083579062224721445417685572465e-02 1.07634727377168594453227257190520e-02
            6.66561537390982167608499686650703e-04 1.00338007966599584985800674985512e-01 2.67977697259694710996136279845814e-02
            2.33630115192034071985860421705183e-03 2.89437843083978907365860777645139e-01 1.99901656639539293536245168070309e-01
            5.23674622175512405157504769448451e-04 1.78616453955091902816576521217939e-01 9.65979284760496922923600493504637e-03
            1.27850641044706782659123867063045e-03 2.86153922631251944608266057912260e-01 4.36162890256223956253478490907582e-02
            5.76882964065722784671730316574667e-04 2.29328895960897921613153016551223e-01 9.63379554976936600119152842580661e-03
            2.11187879154452716468082229539505e-03 3.54212896166471558068167269084370e-01 1.16457002151752600505041357337177e-01
            6.68708757251548313420286273611737e-04 3.44944491544985354369856622724910e-01 9.47687714910445087868051672330694e-03
            1.19071165228710356721875118779508e-03 4.31673230823553855106666787833092e-01 2.79666162959972923462981242437309e-02
            2.14142622600952022005049713015978e-03 2.20797223751839605254687626256782e-01 1.47265844196796946707905817675055e-01
            2.20346979268820853054347552735237e-04 1.53172835097684312266608230856946e-01 1.87056691103424949790268172478136e-03
            5.78226288753999189415702653072060e-05 9.19861676581091788906796580249647e-03 2.06714017852669502614548768804070e-03
            9.12125509019910923717844308811209e-05 2.35452342574103377914607193588381e-02 2.00638084202376830098013549275038e-03
            2.56755792363438451256230088404209e-03 3.45728479718466152892375475857989e-01 1.96591992279532390419660714542260e-01
            1.95252785507732591098462027190408e-03 2.50169308812157320520697112442576e-01 1.07478866102230660728444888718514e-01
            2.64108779916953488077491218888326e-04 2.55549862601632926661920919286786e-01 1.86180935326567915596984725823404e-03
            9.49531737923529016963397619832676e-04 1.03293435262624994086522178804444e-01 5.02072925915596976476074075890210e-02
            1.16702965608184877112363242446236e-03 1.13482779190540200575654239401047e-01 7.90987667568746055657413762673968e-02
            2.41335338989844840583420815249838e-04 2.02001519139572843775098931473622e-01 1.82886968001511268991365888325618e-03
            1.62450972466069874441951803589745e-03 1.55407293341666141728296679502819e-01 9.74984606749395160685267569533607e-02
            6.59883458383826119325599179887831e-04 2.83733125849908784932296157421661e-01 1.02710427379365670796929066455050e-02
            1.98364630522206467858753464739152e-04 1.10137610711961098086675292506698e-01 1.94506677452367052888471121718794e-03
            2.92796720510567231622600647966692e-04 3.11626986396292382508477203373332e-01 2.01271492926212132792973541484116e-03
            7.00387442375592335699896473499848e-04 4.56065627549956553732357633634820e-01 9.87982421462061527472897637380811e-03
            1.85252774407086072362346729214266e-04 7.40590746798509141912703057641920e-02 2.22615254701005607987362999722336e-03
            ];
        
        S2=[
            1.66434758735661434900232613642856e-03 2.56355863384790827463888263082481e-01
            1.82966121231848855986819313290681e-05 1.63057814214604269824338000915986e-03
            8.25698178639569209738557908906387e-04 4.90326869248451202576433161084424e-01
            2.12954833909573520295266391144651e-03 4.22587168507520649640696319693234e-01
            1.21213997870494661138196512695231e-03 4.80575676434030041317413406432024e-01
            1.79135560553105315124977625629299e-03 4.59586771824411877140903470717603e-01
            1.38979940022233134332141357702994e-03 1.13333234135074248372099248172162e-01
            1.49976093457403344359063490642825e-04 1.09390130646629217536780487307624e-02
            1.03393583353988700174186377012120e-03 7.86463619513996653465071062782954e-02
            2.01385549425915546389864729803776e-03 1.70258882364131336473178635060322e-01
            2.28533694456308986153736384494550e-03 1.99633459336817115081785800612124e-01
            2.58267796372693659787134201621939e-03 4.00668932548830014628293838541140e-01
            1.74263167672632809761246175384031e-03 1.40613874056384147159803887916496e-01
            2.92840722695922621096609184121462e-03 3.51190364445415514893511499394663e-01
            5.99474391919224192971926701289931e-04 4.22503625151379735847001484216889e-02
            3.74661027052605310073279953897440e-04 4.98683898312057949375741827680031e-01
            2.88681945179615441141041110029164e-03 2.96479867512970896026303080361686e-01
            ];
        
        S3=[];
        
        
        
    otherwise
        S1=[]; S2=[]; S3=[];
end




pointset_stats=[
    1    1 1.92e-16    0    0
    2    3 4.20e-16    0    0
    3    6 1.25e-15    0    0
    4    6 1.25e-15    0    0
    5    7 7.08e-16    0    0
    6   12 4.30e-16    0    0
    7   15 7.15e-16    0    0
    8   16 9.99e-16    0    0
    9   19 1.12e-15    0    0
    10   25 6.66e-16    0    0
    11   28 5.83e-16    0    0
    12   33 8.47e-16    0    0
    13   37 1.06e-15    0    0
    14   42 1.14e-15    0    0
    15   49 8.50e-16    0    0
    16   55 2.03e-15    0    0
    17   60 9.49e-16    0    0
    18   67 1.50e-15    0    0
    19   73 1.07e-15    0    0
    20   79 1.25e-15    0    0
    21   87 1.42e-15    0    0
    22   96 9.99e-16    0    0
    23  103 2.05e-15    0    0
    24  112 9.85e-16    0    0
    25  120 1.80e-15    0    0
    26  130 1.96e-15    0    0
    27  141 1.11e-15    0    0
    28  150 1.39e-15    0    0
    29  159 1.34e-15    0    0
    30  171 1.64e-15    0    0
    31  181 1.70e-15    0    0
    32  193 1.71e-15    0    0
    33  204 1.45e-15    0    0
    34  214 1.87e-15    0    0
    35  228 1.68e-15    0    0
    36  243 1.18e-15    0    0
    37  252 1.03e-15    0    0
    38  267 1.85e-15    0    0
    39  282 1.58e-15    0    0
    40  295 1.42e-15    0    0
    41  309 1.45e-15    0    0
    42  324 1.44e-15    0    0
    43  339 1.45e-15    0    0
    44  354 1.38e-15    0    0
    45  370 1.52e-15    0    0
    46  385 2.44e-15    0    0
    47  399 1.56e-15    0    0
    48  423 1.49e-15    0    0
    49  435 1.23e-15    0    0
    50  453 1.39e-15    0    0
    ];




