
function [xyw,pointset_stats]=set_xiao_gimbutas_standard(deg)

%--------------------------------------------------------------------
% CUBATURE RULE ON THE SIMPLEX [0 0; 0 1; 1 0] WITH ADE EQUAL TO deg.
% THE CUBATURE RULE IS DESCRIBED IN STANDARD FORM.
%--------------------------------------------------------------------
% INPUT:
% deg: algebraic degree of precision (ADE) of the rule.
%--------------------------------------------------------------------
% OUTPUT:
% xyw: MATRIX OF NODES xyw(:,1:2) AND WEIGHTS xyw(:,3).
%      REMARK: The sum of the weights makes 1/2.
%
% pointset_stats:
%    statistics of the pointset stored in a matrix. Each rows contains
%    degree, cardinality of the full pointset, moment error w.r.t. orthn.
%    Dubiner basis,number of points with negative weights, number of
%    points outside the domain.
%--------------------------------------------------------------------
% SOURCE:
%  H. Xiao, Z. Gimbutas.
%  "A numerical algorithm for the construction of efficient quadrature
%  rules in two and higher dimensions".
%  Computers and Mathematics with Applications 59 (2010) 663-676.
%--------------------------------------------------------------------
%  REMARK.
%  In the original paper, Xiao and Gimbutas proposed three sets with
%  different symmetries, but they do not suggest where to find them in
%  terms of files.
%  We downloaded this file from the url
%          https://cims.nyu.edu/cmcl/quadratures/quadratures.html
%  modifying it to be used directly as cubature rule.
%  It is just one of the three, i.e. the pointset with fully symmetric
%  rules.
%--------------------------------------------------------------------
% | DEG | CARD | W- | PT.O|   M.E.   |Q.|
%--------------------------------------------------------------------
% |   1 |   1 |   0 |   0 | 1.92e-16 |PI|
% |   2 |   3 |   0 |   0 | 4.20e-16 |PI|
% |   3 |   6 |   0 |   0 | 1.25e-15 |PI|
% |   4 |   6 |   0 |   0 | 1.25e-15 |PI|
% |   5 |   7 |   0 |   0 | 7.08e-16 |PI|
% |   6 |  12 |   0 |   0 | 4.30e-16 |PI|
% |   7 |  15 |   0 |   0 | 7.15e-16 |PI|
% |   8 |  16 |   0 |   0 | 9.99e-16 |PI|
% |   9 |  19 |   0 |   0 | 1.12e-15 |PI|
% |  10 |  25 |   0 |   0 | 6.66e-16 |PI|
% |  11 |  28 |   0 |   0 | 5.83e-16 |PI|
% |  12 |  33 |   0 |   0 | 8.47e-16 |PI|
% |  13 |  37 |   0 |   0 | 1.06e-15 |PI|
% |  14 |  42 |   0 |   0 | 1.14e-15 |PI|
% |  15 |  49 |   0 |   0 | 8.50e-16 |PI|
% |  16 |  55 |   0 |   0 | 2.03e-15 |PI|
% |  17 |  60 |   0 |   0 | 9.49e-16 |PI|
% |  18 |  67 |   0 |   0 | 1.50e-15 |PI|
% |  19 |  73 |   0 |   0 | 1.07e-15 |PI|
% |  20 |  79 |   0 |   0 | 1.25e-15 |PI|
% |  21 |  87 |   0 |   0 | 1.42e-15 |PI|
% |  22 |  96 |   0 |   0 | 9.99e-16 |PI|
% |  23 | 103 |   0 |   0 | 2.05e-15 |PI|
% |  24 | 112 |   0 |   0 | 9.85e-16 |PI|
% |  25 | 120 |   0 |   0 | 1.80e-15 |PI|
% |  26 | 130 |   0 |   0 | 1.96e-15 |PI|
% |  27 | 141 |   0 |   0 | 1.11e-15 |PI|
% |  28 | 150 |   0 |   0 | 1.39e-15 |PI|
% |  29 | 159 |   0 |   0 | 1.34e-15 |PI|
% |  30 | 171 |   0 |   0 | 1.64e-15 |PI|
% |  31 | 181 |   0 |   0 | 1.70e-15 |PI|
% |  32 | 193 |   0 |   0 | 1.71e-15 |PI|
% |  33 | 204 |   0 |   0 | 1.45e-15 |PI|
% |  34 | 214 |   0 |   0 | 1.87e-15 |PI|
% |  35 | 228 |   0 |   0 | 1.68e-15 |PI|
% |  36 | 243 |   0 |   0 | 1.18e-15 |PI|
% |  37 | 252 |   0 |   0 | 1.03e-15 |PI|
% |  38 | 267 |   0 |   0 | 1.85e-15 |PI|
% |  39 | 282 |   0 |   0 | 1.58e-15 |PI|
% |  40 | 295 |   0 |   0 | 1.42e-15 |PI|
% |  41 | 309 |   0 |   0 | 1.45e-15 |PI|
% |  42 | 324 |   0 |   0 | 1.44e-15 |PI|
% |  43 | 339 |   0 |   0 | 1.45e-15 |PI|
% |  44 | 354 |   0 |   0 | 1.38e-15 |PI|
% |  45 | 370 |   0 |   0 | 1.52e-15 |PI|
% |  46 | 385 |   0 |   0 | 2.44e-15 |PI|
% |  47 | 399 |   0 |   0 | 1.56e-15 |PI|
% |  48 | 423 |   0 |   0 | 1.49e-15 |PI|
% |  49 | 435 |   0 |   0 | 1.23e-15 |PI|
% |  50 | 453 |   0 |   0 | 1.39e-15 |PI|
%--------------------------------------------------------------------
% LEGEND:
%--------------------------------------------------------------------
% DEG  : algebraic degree of precision
% CARD : cardinality of the pointset
% W-   : negative weights
% PT.O : points outside the simplex [0 0; 1 0; 0 1].
% M.E. : maximum moment error in inf norm.
% Q.: quality of the rule.
%        PI: pos. weights, points in simplex
%        NI: some neg. weights, points in simplex
%        P0: pos. weights, some points out of the simplex
%        N0: some neg. weights, some points out of the simplex
%--------------------------------------------------------------------



switch deg
    
    case   1
        
        % ALG. DEG.:    1
        % PTS CARD.:    1
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.9e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 5.00000000000000000000000000000000e-01
            ];
        
    case   2
        
        % ALG. DEG.:    2
        % PTS CARD.:    3
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.2e-16
        
        xyw=[
            1.66666666666666601903656896865868e-01 1.66666666666666601903656896865868e-01 1.66666666666666657414808128123695e-01
            1.66666666666666601903656896865868e-01 6.66666666666666740681534975010436e-01 1.66666666666666657414808128123695e-01
            6.66666666666666740681534975010436e-01 1.66666666666666601903656896865868e-01 1.66666666666666657414808128123695e-01
            ];
        
    case   3
        
        % ALG. DEG.:    3
        % PTS CARD.:    6
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.2e-15
        
        xyw=[
            4.45948490915964834702123198439949e-01 4.45948490915964834702123198439949e-01 1.11690794839005722027813760632853e-01
            4.45948490915964834702123198439949e-01 1.08103018168070330595753603120102e-01 1.11690794839005722027813760632853e-01
            1.08103018168070330595753603120102e-01 4.45948490915964834702123198439949e-01 1.11690794839005722027813760632853e-01
            9.15762135097706597708366871302132e-02 9.15762135097706597708366871302132e-02 5.49758718276609284481004635836143e-02
            9.15762135097706597708366871302132e-02 8.16847572980458735969477856997401e-01 5.49758718276609284481004635836143e-02
            8.16847572980458735969477856997401e-01 9.15762135097706597708366871302132e-02 5.49758718276609284481004635836143e-02
            ];
        
    case   4
        
        % ALG. DEG.:    4
        % PTS CARD.:    6
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.2e-15
        
        xyw=[
            4.45948490915964834702123198439949e-01 4.45948490915964834702123198439949e-01 1.11690794839005722027813760632853e-01
            4.45948490915964834702123198439949e-01 1.08103018168070330595753603120102e-01 1.11690794839005722027813760632853e-01
            1.08103018168070330595753603120102e-01 4.45948490915964834702123198439949e-01 1.11690794839005722027813760632853e-01
            9.15762135097706597708366871302132e-02 9.15762135097706597708366871302132e-02 5.49758718276609284481004635836143e-02
            9.15762135097706597708366871302132e-02 8.16847572980458735969477856997401e-01 5.49758718276609284481004635836143e-02
            8.16847572980458735969477856997401e-01 9.15762135097706597708366871302132e-02 5.49758718276609284481004635836143e-02
            ];
        
    case   5
        
        % ALG. DEG.:    5
        % PTS CARD.:    7
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 7.1e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 1.12500000000000016653345369377348e-01
            1.01286507323456259621607955523359e-01 1.01286507323456259621607955523359e-01 6.29695902724135836425745083033689e-02
            1.01286507323456259621607955523359e-01 7.97426985353087536267935320211109e-01 6.29695902724135836425745083033689e-02
            7.97426985353087536267935320211109e-01 1.01286507323456259621607955523359e-01 6.29695902724135836425745083033689e-02
            4.70142064105115053962435922585428e-01 4.70142064105115053962435922585428e-01 6.61970763942530959766941123234574e-02
            4.70142064105115053962435922585428e-01 5.97158717897698920751281548291445e-02 6.61970763942530959766941123234574e-02
            5.97158717897698920751281548291445e-02 4.70142064105115053962435922585428e-01 6.61970763942530959766941123234574e-02
            ];
        
    case   6
        
        % ALG. DEG.:    6
        % PTS CARD.:   12
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.3e-16
        
        xyw=[
            2.19429982549782909284630250112968e-01 2.19429982549782909284630250112968e-01 8.56665620764905377804154795740033e-02
            2.19429982549782909284630250112968e-01 5.61140034900434181430739499774063e-01 8.56665620764905377804154795740033e-02
            5.61140034900434181430739499774063e-01 2.19429982549782909284630250112968e-01 8.56665620764905377804154795740033e-02
            4.80137964112215043233078404227854e-01 4.80137964112215043233078404227854e-01 4.03655447965155025902284080530080e-02
            4.80137964112215043233078404227854e-01 3.97240717755699135338431915442925e-02 4.03655447965155025902284080530080e-02
            3.97240717755699135338431915442925e-02 4.80137964112215043233078404227854e-01 4.03655447965155025902284080530080e-02
            1.41619015923968138981337006043759e-01 1.93717243612408114572520645424447e-02 2.03172798968303362776577358772556e-02
            1.41619015923968138981337006043759e-01 8.39009259714791122419796920439694e-01 2.03172798968303362776577358772556e-02
            1.93717243612408114572520645424447e-02 1.41619015923968138981337006043759e-01 2.03172798968303362776577358772556e-02
            1.93717243612408114572520645424447e-02 8.39009259714791122419796920439694e-01 2.03172798968303362776577358772556e-02
            8.39009259714791122419796920439694e-01 1.41619015923968138981337006043759e-01 2.03172798968303362776577358772556e-02
            8.39009259714791122419796920439694e-01 1.93717243612408114572520645424447e-02 2.03172798968303362776577358772556e-02
            ];
        
    case   7
        
        % ALG. DEG.:    7
        % PTS CARD.:   15
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 7.1e-16
        
        xyw=[
            4.73195653689251038986895991911297e-01 4.73195653689251038986895991911297e-01 2.65904166483802266440772399391790e-02
            4.73195653689251038986895991911297e-01 5.36086926214979220262080161774065e-02 2.65904166483802266440772399391790e-02
            5.36086926214979220262080161774065e-02 4.73195653689251038986895991911297e-01 2.65904166483802266440772399391790e-02
            5.77976400545063204172713255957206e-02 5.77976400545063204172713255957206e-02 2.04590851970284336169836336694061e-02
            5.77976400545063204172713255957206e-02 8.84404719890987345287669540994102e-01 2.04590851970284336169836336694061e-02
            8.84404719890987345287669540994102e-01 5.77976400545063204172713255957206e-02 2.04590851970284336169836336694061e-02
            2.41663606397247349155676943155413e-01 2.41663606397247349155676943155413e-01 6.38626242805669236402010824349418e-02
            2.41663606397247349155676943155413e-01 5.16672787205505246177494882431347e-01 6.38626242805669236402010824349418e-02
            5.16672787205505246177494882431347e-01 2.41663606397247349155676943155413e-01 6.38626242805669236402010824349418e-02
            2.59339011865785673549567036388908e-01 4.69712061300855548973132158607768e-02 2.78772702703455436956669899473127e-02
            2.59339011865785673549567036388908e-01 6.93689782004128785430907555564772e-01 2.78772702703455436956669899473127e-02
            4.69712061300855548973132158607768e-02 2.59339011865785673549567036388908e-01 2.78772702703455436956669899473127e-02
            4.69712061300855548973132158607768e-02 6.93689782004128785430907555564772e-01 2.78772702703455436956669899473127e-02
            6.93689782004128785430907555564772e-01 2.59339011865785673549567036388908e-01 2.78772702703455436956669899473127e-02
            6.93689782004128785430907555564772e-01 4.69712061300855548973132158607768e-02 2.78772702703455436956669899473127e-02
            ];
        
    case   8
        
        % ALG. DEG.:    8
        % PTS CARD.:   16
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.0e-15
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 7.21578038388935860680817313550506e-02
            1.70569307751760101954374704291695e-01 1.70569307751760101954374704291695e-01 5.16086852673591223172877562319627e-02
            1.70569307751760101954374704291695e-01 6.58861384496479796091250591416610e-01 5.16086852673591223172877562319627e-02
            6.58861384496479796091250591416610e-01 1.70569307751760101954374704291695e-01 5.16086852673591223172877562319627e-02
            4.59292588292723125142913431773195e-01 4.59292588292723125142913431773195e-01 4.75458171336423096597911808203207e-02
            4.59292588292723125142913431773195e-01 8.14148234145537497141731364536099e-02 4.75458171336423096597911808203207e-02
            8.14148234145537497141731364536099e-02 4.59292588292723125142913431773195e-01 4.75458171336423096597911808203207e-02
            5.05472283170308595012798491552530e-02 5.05472283170308595012798491552530e-02 1.62292488115990396480015078850556e-02
            5.05472283170308595012798491552530e-02 8.98905543365938308753015917318407e-01 1.62292488115990396480015078850556e-02
            8.98905543365938308753015917318407e-01 5.05472283170308595012798491552530e-02 1.62292488115990396480015078850556e-02
            2.63112829634638056841566822185996e-01 8.39477740995767801601701307845360e-03 1.36151570872174963733325370185412e-02
            2.63112829634638056841566822185996e-01 7.28492392955404244325734453013865e-01 1.36151570872174963733325370185412e-02
            8.39477740995767801601701307845360e-03 2.63112829634638056841566822185996e-01 1.36151570872174963733325370185412e-02
            8.39477740995767801601701307845360e-03 7.28492392955404244325734453013865e-01 1.36151570872174963733325370185412e-02
            7.28492392955404244325734453013865e-01 2.63112829634638056841566822185996e-01 1.36151570872174963733325370185412e-02
            7.28492392955404244325734453013865e-01 8.39477740995767801601701307845360e-03 1.36151570872174963733325370185412e-02
            ];
        
    case   9
        
        % ALG. DEG.:    9
        % PTS CARD.:   19
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.1e-15
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 4.85678981413994251270516144813882e-02
            4.89682519198737620236272505280795e-01 4.89682519198737620236272505280795e-01 1.56673501135695392161295558253187e-02
            4.89682519198737620236272505280795e-01 2.06349616025247595274549894384108e-02 1.56673501135695392161295558253187e-02
            2.06349616025247595274549894384108e-02 4.89682519198737620236272505280795e-01 1.56673501135695392161295558253187e-02
            1.88203535619032635839431577551295e-01 1.88203535619032635839431577551295e-01 3.98238694636051382413910459945328e-02
            1.88203535619032635839431577551295e-01 6.23592928761934728321136844897410e-01 3.98238694636051382413910459945328e-02
            6.23592928761934728321136844897410e-01 1.88203535619032635839431577551295e-01 3.98238694636051382413910459945328e-02
            4.37089591492936579975037147960393e-01 4.37089591492936579975037147960393e-01 3.89137705023871391385448248456669e-02
            4.37089591492936579975037147960393e-01 1.25820817014126840049925704079214e-01 3.89137705023871391385448248456669e-02
            1.25820817014126840049925704079214e-01 4.37089591492936579975037147960393e-01 3.89137705023871391385448248456669e-02
            4.47295133944525663549640626115433e-02 4.47295133944525663549640626115433e-02 1.27888378293490173609470161864010e-02
            4.47295133944525663549640626115433e-02 9.10540973211094839534496259148000e-01 1.27888378293490173609470161864010e-02
            9.10540973211094839534496259148000e-01 4.47295133944525663549640626115433e-02 1.27888378293490173609470161864010e-02
            2.21962989160765705731748198559217e-01 3.68384120547363136388696602807613e-02 2.16417696886446880855103813701135e-02
            2.21962989160765705731748198559217e-01 7.41198598784498008384957756788936e-01 2.16417696886446880855103813701135e-02
            3.68384120547363136388696602807613e-02 2.21962989160765705731748198559217e-01 2.16417696886446880855103813701135e-02
            3.68384120547363136388696602807613e-02 7.41198598784498008384957756788936e-01 2.16417696886446880855103813701135e-02
            7.41198598784498008384957756788936e-01 2.21962989160765705731748198559217e-01 2.16417696886446880855103813701135e-02
            7.41198598784498008384957756788936e-01 3.68384120547363136388696602807613e-02 2.16417696886446880855103813701135e-02
            ];
        
    case  10
        
        % ALG. DEG.:   10
        % PTS CARD.:   25
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.7e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 4.18074371869869704232414164835063e-02
            4.95173459801170445793161434266949e-01 4.95173459801170445793161434266949e-01 4.89629524920915260716158812215326e-03
            4.95173459801170445793161434266949e-01 9.65308039765910841367713146610186e-03 4.89629524920915260716158812215326e-03
            9.65308039765910841367713146610186e-03 4.95173459801170445793161434266949e-01 4.89629524920915260716158812215326e-03
            1.91394152428411012945019820108428e-02 1.91394152428411012945019820108428e-02 3.19267961505932847121602691231601e-03
            1.91394152428411012945019820108428e-02 9.61721169514317741899844804720487e-01 3.19267961505932847121602691231601e-03
            9.61721169514317741899844804720487e-01 1.91394152428411012945019820108428e-02 3.19267961505932847121602691231601e-03
            1.84485012685246418939755130850244e-01 1.84485012685246418939755130850244e-01 3.93168848731886497183118933662627e-02
            1.84485012685246418939755130850244e-01 6.31029974629507162120489738299511e-01 3.93168848731886497183118933662627e-02
            6.31029974629507162120489738299511e-01 1.84485012685246418939755130850244e-01 3.93168848731886497183118933662627e-02
            4.28234820943718841323999413361889e-01 4.28234820943718841323999413361889e-01 3.76236639842720058113378911457403e-02
            4.28234820943718841323999413361889e-01 1.43530358112562317352001173276221e-01 3.76236639842720058113378911457403e-02
            1.43530358112562317352001173276221e-01 4.28234820943718841323999413361889e-01 3.76236639842720058113378911457403e-02
            1.33734755100869129318397199313040e-01 3.47236204823274940967081647613668e-02 1.44811407316281746948671838026712e-02
            1.33734755100869129318397199313040e-01 8.31541624416803348829319020296680e-01 1.44811407316281746948671838026712e-02
            3.47236204823274940967081647613668e-02 1.33734755100869129318397199313040e-01 1.44811407316281746948671838026712e-02
            3.47236204823274940967081647613668e-02 8.31541624416803348829319020296680e-01 1.44811407316281746948671838026712e-02
            8.31541624416803348829319020296680e-01 1.33734755100869129318397199313040e-01 1.44811407316281746948671838026712e-02
            8.31541624416803348829319020296680e-01 3.47236204823274940967081647613668e-02 1.44811407316281746948671838026712e-02
            3.26693136281336893311788571736543e-01 3.75827273411917031631368502075929e-02 1.93695245430094559191402936448867e-02
            3.26693136281336893311788571736543e-01 6.35724136377471382708392866334179e-01 1.93695245430094559191402936448867e-02
            3.75827273411917031631368502075929e-02 3.26693136281336893311788571736543e-01 1.93695245430094559191402936448867e-02
            3.75827273411917031631368502075929e-02 6.35724136377471382708392866334179e-01 1.93695245430094559191402936448867e-02
            6.35724136377471382708392866334179e-01 3.26693136281336893311788571736543e-01 1.93695245430094559191402936448867e-02
            6.35724136377471382708392866334179e-01 3.75827273411917031631368502075929e-02 1.93695245430094559191402936448867e-02
            ];
        
    case  11
        
        % ALG. DEG.:   11
        % PTS CARD.:   28
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.8e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 4.07225673546756164467019800667913e-02
            3.08468956355878905217782204317700e-02 3.08468956355878905217782204317700e-02 6.12464847535397743233387402028711e-03
            3.08468956355878905217782204317700e-02 9.38306208728824198139761847414775e-01 6.12464847535397743233387402028711e-03
            9.38306208728824198139761847414775e-01 3.08468956355878905217782204317700e-02 6.12464847535397743233387402028711e-03
            4.98780165178460743735655569253140e-01 4.98780165178460743735655569253140e-01 6.23274593694068696630994352858579e-03
            4.98780165178460743735655569253140e-01 2.43966964307851252868886149371974e-03 6.23274593694068696630994352858579e-03
            2.43966964307851252868886149371974e-03 4.98780165178460743735655569253140e-01 6.23274593694068696630994352858579e-03
            1.13207827286693873958256517653354e-01 1.13207827286693873958256517653354e-01 2.00646211906541430125550107277377e-02
            1.13207827286693873958256517653354e-01 7.73584345426612252083486964693293e-01 2.00646211906541430125550107277377e-02
            7.73584345426612252083486964693293e-01 1.13207827286693873958256517653354e-01 2.00646211906541430125550107277377e-02
            4.36655016393176054201319402636727e-01 4.36655016393176054201319402636727e-01 3.15474360799493230844703361981374e-02
            4.36655016393176054201319402636727e-01 1.26689967213647891597361194726545e-01 3.15474360799493230844703361981374e-02
            1.26689967213647891597361194726545e-01 4.36655016393176054201319402636727e-01 3.15474360799493230844703361981374e-02
            2.14483458619269262701578782071010e-01 2.14483458619269262701578782071010e-01 3.39225538718475536081875532090635e-02
            2.14483458619269262701578782071010e-01 5.71033082761461474596842435857980e-01 3.39225538718475536081875532090635e-02
            5.71033082761461474596842435857980e-01 2.14483458619269262701578782071010e-01 3.39225538718475536081875532090635e-02
            1.59303619837693477556683774309931e-01 1.43666625695556288838661274098740e-02 7.27881166890461880902840619000926e-03
            1.59303619837693477556683774309931e-01 8.26329717592750867538597958628088e-01 7.27881166890461880902840619000926e-03
            1.43666625695556288838661274098740e-02 1.59303619837693477556683774309931e-01 7.27881166890461880902840619000926e-03
            1.43666625695556288838661274098740e-02 8.26329717592750867538597958628088e-01 7.27881166890461880902840619000926e-03
            8.26329717592750867538597958628088e-01 1.59303619837693477556683774309931e-01 7.27881166890461880902840619000926e-03
            8.26329717592750867538597958628088e-01 1.43666625695556288838661274098740e-02 7.27881166890461880902840619000926e-03
            3.10631216313463132117078657756792e-01 4.76640669721508022926848013867129e-02 2.03214243279432223154845615908926e-02
            3.10631216313463132117078657756792e-01 6.41704716714386114162493868207093e-01 2.03214243279432223154845615908926e-02
            4.76640669721508022926848013867129e-02 3.10631216313463132117078657756792e-01 2.03214243279432223154845615908926e-02
            4.76640669721508022926848013867129e-02 6.41704716714386114162493868207093e-01 2.03214243279432223154845615908926e-02
            6.41704716714386114162493868207093e-01 3.10631216313463132117078657756792e-01 2.03214243279432223154845615908926e-02
            6.41704716714386114162493868207093e-01 4.76640669721508022926848013867129e-02 2.03214243279432223154845615908926e-02
            ];
        
    case  12
        
        % ALG. DEG.:   12
        % PTS CARD.:   33
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 8.5e-16
        
        xyw=[
            2.71462507014926024417889038886642e-01 2.71462507014926024417889038886642e-01 3.12706065979513961328173365927796e-02
            2.71462507014926024417889038886642e-01 4.57074985970147951164221922226716e-01 3.12706065979513961328173365927796e-02
            4.57074985970147951164221922226716e-01 2.71462507014926024417889038886642e-01 3.12706065979513961328173365927796e-02
            1.09257827659354198046592898663221e-01 1.09257827659354198046592898663221e-01 1.42430260344387788623565782586411e-02
            1.09257827659354198046592898663221e-01 7.81484344681291576151238587044645e-01 1.42430260344387788623565782586411e-02
            7.81484344681291576151238587044645e-01 1.09257827659354198046592898663221e-01 1.42430260344387788623565782586411e-02
            4.40111648658593090921442581020528e-01 4.40111648658593090921442581020528e-01 2.49591674640304850285588145197835e-02
            4.40111648658593090921442581020528e-01 1.19776702682813818157114837958943e-01 2.49591674640304850285588145197835e-02
            1.19776702682813818157114837958943e-01 4.40111648658593090921442581020528e-01 2.49591674640304850285588145197835e-02
            4.88203750945541525840809526926023e-01 4.88203750945541525840809526926023e-01 1.21334190407260227029251353769723e-02
            4.88203750945541525840809526926023e-01 2.35924981089169483183809461479541e-02 1.21334190407260227029251353769723e-02
            2.35924981089169483183809461479541e-02 4.88203750945541525840809526926023e-01 1.21334190407260227029251353769723e-02
            2.46463634363354618106445315106612e-02 2.46463634363354618106445315106612e-02 3.96582125498682117048510065160372e-03
            2.46463634363354618106445315106612e-02 9.50707273127329055562029225256993e-01 3.96582125498682117048510065160372e-03
            9.50707273127329055562029225256993e-01 2.46463634363354618106445315106612e-02 3.96582125498682117048510065160372e-03
            2.55454228638517299998511589365080e-01 1.16296019677926645852217291121633e-01 2.16136818297071146843801869863455e-02
            2.55454228638517299998511589365080e-01 6.28249751683556012515907696069917e-01 2.16136818297071146843801869863455e-02
            1.16296019677926645852217291121633e-01 2.55454228638517299998511589365080e-01 2.16136818297071146843801869863455e-02
            1.16296019677926645852217291121633e-01 6.28249751683556012515907696069917e-01 2.16136818297071146843801869863455e-02
            6.28249751683556012515907696069917e-01 2.55454228638517299998511589365080e-01 2.16136818297071146843801869863455e-02
            6.28249751683556012515907696069917e-01 1.16296019677926645852217291121633e-01 2.16136818297071146843801869863455e-02
            1.27279717233589356739642539650958e-01 2.13824902561706338410818517559164e-02 7.54183878825572407561539378662019e-03
            1.27279717233589356739642539650958e-01 8.51337792510239999010934752732283e-01 7.54183878825572407561539378662019e-03
            2.13824902561706338410818517559164e-02 1.27279717233589356739642539650958e-01 7.54183878825572407561539378662019e-03
            2.13824902561706338410818517559164e-02 8.51337792510239999010934752732283e-01 7.54183878825572407561539378662019e-03
            8.51337792510239999010934752732283e-01 1.27279717233589356739642539650958e-01 7.54183878825572407561539378662019e-03
            8.51337792510239999010934752732283e-01 2.13824902561706338410818517559164e-02 7.54183878825572407561539378662019e-03
            2.91655679738340944950891753251199e-01 2.30341563552671768633217652677558e-02 1.08917925193037831016473759859764e-02
            2.91655679738340944950891753251199e-01 6.85310163906391878185786481481045e-01 1.08917925193037831016473759859764e-02
            2.30341563552671768633217652677558e-02 2.91655679738340944950891753251199e-01 1.08917925193037831016473759859764e-02
            2.30341563552671768633217652677558e-02 6.85310163906391878185786481481045e-01 1.08917925193037831016473759859764e-02
            6.85310163906391878185786481481045e-01 2.91655679738340944950891753251199e-01 1.08917925193037831016473759859764e-02
            6.85310163906391878185786481481045e-01 2.30341563552671768633217652677558e-02 1.08917925193037831016473759859764e-02
            ];
        
    case  13
        
        % ALG. DEG.:   13
        % PTS CARD.:   37
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.1e-15
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 2.58113233321454194069133336597588e-02
            4.96135894741046079570168103600736e-01 4.96135894741046079570168103600736e-01 4.97073818053629656876779563390301e-03
            4.96135894741046079570168103600736e-01 7.72821051790784085966379279852845e-03 4.97073818053629656876779563390301e-03
            7.72821051790784085966379279852845e-03 4.96135894741046079570168103600736e-01 4.97073818053629656876779563390301e-03
            4.69608689653491895299453062762041e-01 4.69608689653491895299453062762041e-01 1.63906208018614919785793659912088e-02
            4.69608689653491895299453062762041e-01 6.07826206930162094010938744759187e-02 1.63906208018614919785793659912088e-02
            6.07826206930162094010938744759187e-02 4.69608689653491895299453062762041e-01 1.63906208018614919785793659912088e-02
            2.31110284949082200878223147810786e-01 2.31110284949082200878223147810786e-01 2.30312047963891312896134166976481e-02
            2.31110284949082200878223147810786e-01 5.37779430101835598243553704378428e-01 2.30312047963891312896134166976481e-02
            5.37779430101835598243553704378428e-01 2.31110284949082200878223147810786e-01 2.30312047963891312896134166976481e-02
            4.14477570279054519097883257927606e-01 4.14477570279054519097883257927606e-01 2.34735477710776054005581414685366e-02
            4.14477570279054519097883257927606e-01 1.71044859441890961804233484144788e-01 2.34735477710776054005581414685366e-02
            1.71044859441890961804233484144788e-01 4.14477570279054519097883257927606e-01 2.34735477710776054005581414685366e-02
            1.13559912572133075991942519067379e-01 1.13559912572133075991942519067379e-01 1.54515489878799035933232275397131e-02
            1.13559912572133075991942519067379e-01 7.72880174855733903527266193123069e-01 1.54515489878799035933232275397131e-02
            7.72880174855733903527266193123069e-01 1.13559912572133075991942519067379e-01 1.54515489878799035933232275397131e-02
            2.48959314912162443800269784333068e-02 2.48959314912162443800269784333068e-02 4.01469989762921232456793774190373e-03
            2.48959314912162443800269784333068e-02 9.50208137017567455728794811875559e-01 4.01469989762921232456793774190373e-03
            9.50208137017567455728794811875559e-01 2.48959314912162443800269784333068e-02 4.01469989762921232456793774190373e-03
            2.92078688576636380513917856660555e-01 1.89880043837590481436983225194126e-02 9.06274932310044412098193333804375e-03
            2.92078688576636380513917856660555e-01 6.88933307039604536647914301283890e-01 9.06274932310044412098193333804375e-03
            1.89880043837590481436983225194126e-02 2.92078688576636380513917856660555e-01 9.06274932310044412098193333804375e-03
            1.89880043837590481436983225194126e-02 6.88933307039604536647914301283890e-01 9.06274932310044412098193333804375e-03
            6.88933307039604536647914301283890e-01 2.92078688576636380513917856660555e-01 9.06274932310044412098193333804375e-03
            6.88933307039604536647914301283890e-01 1.89880043837590481436983225194126e-02 9.06274932310044412098193333804375e-03
            2.66745253310351149167445328203030e-01 9.77360310660165543250599284874625e-02 1.86059802286307751639782281927182e-02
            2.66745253310351149167445328203030e-01 6.35518715623632268751919127680594e-01 1.86059802286307751639782281927182e-02
            9.77360310660165543250599284874625e-02 2.66745253310351149167445328203030e-01 1.86059802286307751639782281927182e-02
            9.77360310660165543250599284874625e-02 6.35518715623632268751919127680594e-01 1.86059802286307751639782281927182e-02
            6.35518715623632268751919127680594e-01 2.66745253310351149167445328203030e-01 1.86059802286307751639782281927182e-02
            6.35518715623632268751919127680594e-01 9.77360310660165543250599284874625e-02 1.86059802286307751639782281927182e-02
            1.26799775783837304077295016213611e-01 2.19663442065292507932028343020647e-02 7.69653634189109130819828052949561e-03
            1.26799775783837304077295016213611e-01 8.51233880009633514518441188556608e-01 7.69653634189109130819828052949561e-03
            2.19663442065292507932028343020647e-02 1.26799775783837304077295016213611e-01 7.69653634189109130819828052949561e-03
            2.19663442065292507932028343020647e-02 8.51233880009633514518441188556608e-01 7.69653634189109130819828052949561e-03
            8.51233880009633514518441188556608e-01 1.26799775783837304077295016213611e-01 7.69653634189109130819828052949561e-03
            8.51233880009633514518441188556608e-01 2.19663442065292507932028343020647e-02 7.69653634189109130819828052949561e-03
            ];
        
    case  14
        
        % ALG. DEG.:   14
        % PTS CARD.:   42
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.1e-15
        
        xyw=[
            4.17644719340453884903752168611391e-01 4.17644719340453884903752168611391e-01 1.63941767720626845050713171758616e-02
            4.17644719340453884903752168611391e-01 1.64710561319092230192495662777219e-01 1.63941767720626845050713171758616e-02
            1.64710561319092230192495662777219e-01 4.17644719340453884903752168611391e-01 1.63941767720626845050713171758616e-02
            6.17998830908724830490719170938974e-02 6.17998830908724830490719170938974e-02 7.21684983488833729087241763977545e-03
            6.17998830908724830490719170938974e-02 8.76400233818255047779643973626662e-01 7.21684983488833729087241763977545e-03
            8.76400233818255047779643973626662e-01 6.17998830908724830490719170938974e-02 7.21684983488833729087241763977545e-03
            2.73477528308838591097895687198616e-01 2.73477528308838591097895687198616e-01 2.58870522536458064211117857666977e-02
            2.73477528308838591097895687198616e-01 4.53044943382322817804208625602769e-01 2.58870522536458064211117857666977e-02
            4.53044943382322817804208625602769e-01 2.73477528308838591097895687198616e-01 2.58870522536458064211117857666977e-02
            1.77205532412543387277281681235763e-01 1.77205532412543387277281681235763e-01 2.10812943684965219126592472775883e-02
            1.77205532412543387277281681235763e-01 6.45588935174913225445436637528474e-01 2.10812943684965219126592472775883e-02
            6.45588935174913225445436637528474e-01 1.77205532412543387277281681235763e-01 2.10812943684965219126592472775883e-02
            1.93909612487008810310928907938433e-02 1.93909612487008810310928907938433e-02 2.46170180120004224166851969357594e-03
            1.93909612487008810310928907938433e-02 9.61218077502598244876708122319542e-01 2.46170180120004224166851969357594e-03
            9.61218077502598244876708122319542e-01 1.93909612487008810310928907938433e-02 2.46170180120004224166851969357594e-03
            4.88963910362178622026618768359185e-01 4.88963910362178622026618768359185e-01 1.09417906847144516535585978544987e-02
            4.88963910362178622026618768359185e-01 2.20721792756427559467624632816296e-02 1.09417906847144516535585978544987e-02
            2.20721792756427559467624632816296e-02 4.88963910362178622026618768359185e-01 1.09417906847144516535585978544987e-02
            2.98372882136257677743174099305179e-01 1.46469500556544764136024383560652e-02 7.21815405676692369019153971976266e-03
            2.98372882136257677743174099305179e-01 6.86980167808087793801519183034543e-01 7.21815405676692369019153971976266e-03
            1.46469500556544764136024383560652e-02 2.98372882136257677743174099305179e-01 7.21815405676692369019153971976266e-03
            1.46469500556544764136024383560652e-02 6.86980167808087793801519183034543e-01 7.21815405676692369019153971976266e-03
            6.86980167808087793801519183034543e-01 2.98372882136257677743174099305179e-01 7.21815405676692369019153971976266e-03
            6.86980167808087793801519183034543e-01 1.46469500556544764136024383560652e-02 7.21815405676692369019153971976266e-03
            3.36861459796345019679364440889913e-01 9.29162493569718889174779974382545e-02 1.92857553935303488445995867550664e-02
            3.36861459796345019679364440889913e-01 5.70222290846683077525369753857376e-01 1.92857553935303488445995867550664e-02
            9.29162493569718889174779974382545e-02 3.36861459796345019679364440889913e-01 1.92857553935303488445995867550664e-02
            9.29162493569718889174779974382545e-02 5.70222290846683077525369753857376e-01 1.92857553935303488445995867550664e-02
            5.70222290846683077525369753857376e-01 3.36861459796345019679364440889913e-01 1.92857553935303488445995867550664e-02
            5.70222290846683077525369753857376e-01 9.29162493569718889174779974382545e-02 1.92857553935303488445995867550664e-02
            1.72266687821355540810586148836592e-01 5.71247574036480090797773812028026e-02 1.23328766062818454690708236398677e-02
            1.72266687821355540810586148836592e-01 7.70608554774996457048530373867834e-01 1.23328766062818454690708236398677e-02
            5.71247574036480090797773812028026e-02 1.72266687821355540810586148836592e-01 1.23328766062818454690708236398677e-02
            5.71247574036480090797773812028026e-02 7.70608554774996457048530373867834e-01 1.23328766062818454690708236398677e-02
            7.70608554774996457048530373867834e-01 1.72266687821355540810586148836592e-01 1.23328766062818454690708236398677e-02
            7.70608554774996457048530373867834e-01 5.71247574036480090797773812028026e-02 1.23328766062818454690708236398677e-02
            1.18974497696956824088765358737874e-01 1.26833093287207670021854433173303e-03 2.50511441925033726332849859375074e-03
            1.18974497696956824088765358737874e-01 8.79757171370171175972529908904107e-01 2.50511441925033726332849859375074e-03
            1.26833093287207670021854433173303e-03 1.18974497696956824088765358737874e-01 2.50511441925033726332849859375074e-03
            1.26833093287207670021854433173303e-03 8.79757171370171175972529908904107e-01 2.50511441925033726332849859375074e-03
            8.79757171370171175972529908904107e-01 1.18974497696956824088765358737874e-01 2.50511441925033726332849859375074e-03
            8.79757171370171175972529908904107e-01 1.26833093287207670021854433173303e-03 2.50511441925033726332849859375074e-03
            ];
        
    case  15
        
        % ALG. DEG.:   15
        % PTS CARD.:   49
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 8.5e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 1.48652098740356657796679584748745e-02
            1.29978229933077754632009259694314e-01 1.29978229933077754632009259694314e-01 3.69875203352305111786924030070622e-03
            1.29978229933077754632009259694314e-01 7.40043540133844546247132711869199e-01 3.69875203352305111786924030070622e-03
            7.40043540133844546247132711869199e-01 1.29978229933077754632009259694314e-01 3.69875203352305111786924030070622e-03
            4.60076949297059667465248367079766e-01 4.60076949297059667465248367079766e-01 1.07970439682192297092822030890602e-02
            4.60076949297059667465248367079766e-01 7.98461014058806650695032658404671e-02 1.07970439682192297092822030890602e-02
            7.98461014058806650695032658404671e-02 4.60076949297059667465248367079766e-01 1.07970439682192297092822030890602e-02
            4.91685816630297201079713431681739e-01 4.91685816630297201079713431681739e-01 7.91613817501090125117180207325873e-03
            4.91685816630297201079713431681739e-01 1.66283667394055978405731366365217e-02 7.91613817501090125117180207325873e-03
            1.66283667394055978405731366365217e-02 4.91685816630297201079713431681739e-01 7.91613817501090125117180207325873e-03
            2.21532340795141918787791723843839e-01 2.21532340795141918787791723843839e-01 2.31436430525990485818077502244705e-02
            2.21532340795141918787791723843839e-01 5.56935318409716106913265321054496e-01 2.31436430525990485818077502244705e-02
            5.56935318409716106913265321054496e-01 2.21532340795141918787791723843839e-01 2.31436430525990485818077502244705e-02
            3.96933737409060571899033220688580e-01 3.96933737409060571899033220688580e-01 2.31680206956036242871288521882889e-02
            3.96933737409060571899033220688580e-01 2.06132525181878856201933558622841e-01 2.31680206956036242871288521882889e-02
            2.06132525181878856201933558622841e-01 3.96933737409060571899033220688580e-01 2.31680206956036242871288521882889e-02
            5.63419176960999884551384297992627e-02 5.63419176960999884551384297992627e-02 7.54223712379853677284513580048042e-03
            5.63419176960999884551384297992627e-02 8.87316164607800050845298756030388e-01 7.54223712379853677284513580048042e-03
            8.87316164607800050845298756030388e-01 5.63419176960999884551384297992627e-02 7.54223712379853677284513580048042e-03
            1.82321783407191295722782342636492e-01 8.45942214821918392342325887511834e-02 1.21150043915628086166336174755997e-02
            1.82321783407191295722782342636492e-01 7.33083995110616837287409452983411e-01 1.21150043915628086166336174755997e-02
            8.45942214821918392342325887511834e-02 1.82321783407191295722782342636492e-01 1.21150043915628086166336174755997e-02
            8.45942214821918392342325887511834e-02 7.33083995110616837287409452983411e-01 1.21150043915628086166336174755997e-02
            7.33083995110616837287409452983411e-01 1.82321783407191295722782342636492e-01 1.21150043915628086166336174755997e-02
            7.33083995110616837287409452983411e-01 8.45942214821918392342325887511834e-02 1.21150043915628086166336174755997e-02
            1.50200384065238723430013578763464e-01 1.60270897863454761522739033807738e-02 5.61425214943903159309224193407317e-03
            1.50200384065238723430013578763464e-01 8.33772526148415793478818613948533e-01 5.61425214943903159309224193407317e-03
            1.60270897863454761522739033807738e-02 1.50200384065238723430013578763464e-01 5.61425214943903159309224193407317e-03
            1.60270897863454761522739033807738e-02 8.33772526148415793478818613948533e-01 5.61425214943903159309224193407317e-03
            8.33772526148415793478818613948533e-01 1.50200384065238723430013578763464e-01 5.61425214943903159309224193407317e-03
            8.33772526148415793478818613948533e-01 1.60270897863454761522739033807738e-02 5.61425214943903159309224193407317e-03
            3.23111315163712664588757661476848e-01 9.76504424302423873482936755863193e-02 1.55376102352554803370798808259678e-02
            3.23111315163712664588757661476848e-01 5.79238242406045045207463317638030e-01 1.55376102352554803370798808259678e-02
            9.76504424302423873482936755863193e-02 3.23111315163712664588757661476848e-01 1.55376102352554803370798808259678e-02
            9.76504424302423873482936755863193e-02 5.79238242406045045207463317638030e-01 1.55376102352554803370798808259678e-02
            5.79238242406045045207463317638030e-01 3.23111315163712664588757661476848e-01 1.55376102352554803370798808259678e-02
            5.79238242406045045207463317638030e-01 9.76504424302423873482936755863193e-02 1.55376102352554803370798808259678e-02
            3.07947681483672908164805903652450e-01 1.84542519046331753018730381654677e-02 8.21838104641394937011078525301855e-03
            3.07947681483672908164805903652450e-01 6.73598066611693968575025337486295e-01 8.21838104641394937011078525301855e-03
            1.84542519046331753018730381654677e-02 3.07947681483672908164805903652450e-01 8.21838104641394937011078525301855e-03
            1.84542519046331753018730381654677e-02 6.73598066611693968575025337486295e-01 8.21838104641394937011078525301855e-03
            6.73598066611693968575025337486295e-01 3.07947681483672908164805903652450e-01 8.21838104641394937011078525301855e-03
            6.73598066611693968575025337486295e-01 1.84542519046331753018730381654677e-02 8.21838104641394937011078525301855e-03
            3.80352293011092870700373680392659e-02 1.11353527401374216738993983000228e-03 1.23763300727895838790337812440612e-03
            3.80352293011092870700373680392659e-02 9.60851235424877048174607807595748e-01 1.23763300727895838790337812440612e-03
            1.11353527401374216738993983000228e-03 3.80352293011092870700373680392659e-02 1.23763300727895838790337812440612e-03
            1.11353527401374216738993983000228e-03 9.60851235424877048174607807595748e-01 1.23763300727895838790337812440612e-03
            9.60851235424877048174607807595748e-01 3.80352293011092870700373680392659e-02 1.23763300727895838790337812440612e-03
            9.60851235424877048174607807595748e-01 1.11353527401374216738993983000228e-03 1.23763300727895838790337812440612e-03
            ];
        
    case  16
        
        % ALG. DEG.:   16
        % PTS CARD.:   55
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 2.0e-15
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 2.31139551570956960913072464336437e-02
            6.66744722402381612136323951744998e-02 6.66744722402381612136323951744998e-02 6.21271279778051049857978682666726e-03
            6.66744722402381612136323951744998e-02 8.66651055519523705328310825279914e-01 6.21271279778051049857978682666726e-03
            8.66651055519523705328310825279914e-01 6.66744722402381612136323951744998e-02 6.21271279778051049857978682666726e-03
            2.41321680701378271649559792422224e-01 2.41321680701378271649559792422224e-01 2.05920205348962929992051584804358e-02
            2.41321680701378271649559792422224e-01 5.17356638597243456700880415155552e-01 2.05920205348962929992051584804358e-02
            5.17356638597243456700880415155552e-01 2.41321680701378271649559792422224e-01 2.05920205348962929992051584804358e-02
            4.12798095955223653952259610377951e-01 4.12798095955223653952259610377951e-01 2.04926098934077038915546609132434e-02
            4.12798095955223653952259610377951e-01 1.74403808089552692095480779244099e-01 2.04926098934077038915546609132434e-02
            1.74403808089552692095480779244099e-01 4.12798095955223653952259610377951e-01 2.04926098934077038915546609132434e-02
            1.50063736587034979308441506873351e-01 1.50063736587034979308441506873351e-01 1.43917483513744687978475056411298e-02
            1.50063736587034979308441506873351e-01 6.99872526825930041383116986253299e-01 1.43917483513744687978475056411298e-02
            6.99872526825930041383116986253299e-01 1.50063736587034979308441506873351e-01 1.43917483513744687978475056411298e-02
            4.69548030996684961912990274868207e-01 4.69548030996684961912990274868207e-01 1.35468347338552393827049158403497e-02
            4.69548030996684961912990274868207e-01 6.09039380066300761740194502635859e-02 1.35468347338552393827049158403497e-02
            6.09039380066300761740194502635859e-02 4.69548030996684961912990274868207e-01 1.35468347338552393827049158403497e-02
            1.70416294057182740595557390861359e-02 1.70416294057182740595557390861359e-02 1.89456761913211297818082634591974e-03
            1.70416294057182740595557390861359e-02 9.65916741188563410247525098384358e-01 1.89456761913211297818082634591974e-03
            9.65916741188563410247525098384358e-01 1.70416294057182740595557390861359e-02 1.89456761913211297818082634591974e-03
            4.13769485827085170548400583356852e-01 9.66495440366025790335235967631888e-03 4.09110527661107285096742103291945e-03
            4.13769485827085170548400583356852e-01 5.76565559769254609712163528456585e-01 4.09110527661107285096742103291945e-03
            9.66495440366025790335235967631888e-03 4.13769485827085170548400583356852e-01 4.09110527661107285096742103291945e-03
            9.66495440366025790335235967631888e-03 5.76565559769254609712163528456585e-01 4.09110527661107285096742103291945e-03
            5.76565559769254609712163528456585e-01 4.13769485827085170548400583356852e-01 4.09110527661107285096742103291945e-03
            5.76565559769254609712163528456585e-01 9.66495440366025790335235967631888e-03 4.09110527661107285096742103291945e-03
            3.04179448229479731935498421080410e-01 3.03059433551863786904245756659293e-02 6.99180356232678972688843188620922e-03
            3.04179448229479731935498421080410e-01 6.65514608415333941415781282557873e-01 6.99180356232678972688843188620922e-03
            3.03059433551863786904245756659293e-02 3.04179448229479731935498421080410e-01 6.99180356232678972688843188620922e-03
            3.03059433551863786904245756659293e-02 6.65514608415333941415781282557873e-01 6.99180356232678972688843188620922e-03
            6.65514608415333941415781282557873e-01 3.04179448229479731935498421080410e-01 6.99180356232678972688843188620922e-03
            6.65514608415333941415781282557873e-01 3.03059433551863786904245756659293e-02 6.99180356232678972688843188620922e-03
            8.96090890227058400885340461172746e-02 1.08129727761037564698343160785043e-02 2.87593498524858250109392798776753e-03
            8.96090890227058400885340461172746e-02 8.99577938201190474565294152853312e-01 2.87593498524858250109392798776753e-03
            1.08129727761037564698343160785043e-02 8.96090890227058400885340461172746e-02 2.87593498524858250109392798776753e-03
            1.08129727761037564698343160785043e-02 8.99577938201190474565294152853312e-01 2.87593498524858250109392798776753e-03
            8.99577938201190474565294152853312e-01 8.96090890227058400885340461172746e-02 2.87593498524858250109392798776753e-03
            8.99577938201190474565294152853312e-01 1.08129727761037564698343160785043e-02 2.87593498524858250109392798776753e-03
            2.96615372400382937279061934532365e-01 1.06653160536148483572560508036986e-01 1.58230308409916392564653619956516e-02
            2.96615372400382937279061934532365e-01 5.96731467063468579148377557430649e-01 1.58230308409916392564653619956516e-02
            1.06653160536148483572560508036986e-01 2.96615372400382937279061934532365e-01 1.58230308409916392564653619956516e-02
            1.06653160536148483572560508036986e-01 5.96731467063468579148377557430649e-01 1.58230308409916392564653619956516e-02
            5.96731467063468579148377557430649e-01 2.96615372400382937279061934532365e-01 1.58230308409916392564653619956516e-02
            5.96731467063468579148377557430649e-01 1.06653160536148483572560508036986e-01 1.58230308409916392564653619956516e-02
            1.69763355150289729778023684048094e-01 5.13543153440131280862424034694413e-02 8.82654052355165245902224313567785e-03
            1.69763355150289729778023684048094e-01 7.78882329505697135196840008575236e-01 8.82654052355165245902224313567785e-03
            5.13543153440131280862424034694413e-02 1.69763355150289729778023684048094e-01 8.82654052355165245902224313567785e-03
            5.13543153440131280862424034694413e-02 7.78882329505697135196840008575236e-01 8.82654052355165245902224313567785e-03
            7.78882329505697135196840008575236e-01 1.69763355150289729778023684048094e-01 8.82654052355165245902224313567785e-03
            7.78882329505697135196840008575236e-01 5.13543153440131280862424034694413e-02 8.82654052355165245902224313567785e-03
            2.14048779925847282301631935297337e-01 3.69694270735561368937394099987159e-03 2.30734531986456943564167687554800e-03
            2.14048779925847282301631935297337e-01 7.82254277366797090564887184882537e-01 2.30734531986456943564167687554800e-03
            3.69694270735561368937394099987159e-03 2.14048779925847282301631935297337e-01 2.30734531986456943564167687554800e-03
            3.69694270735561368937394099987159e-03 7.82254277366797090564887184882537e-01 2.30734531986456943564167687554800e-03
            7.82254277366797090564887184882537e-01 2.14048779925847282301631935297337e-01 2.30734531986456943564167687554800e-03
            7.82254277366797090564887184882537e-01 3.69694270735561368937394099987159e-03 2.30734531986456943564167687554800e-03
            ];
        
    case  17
        
        % ALG. DEG.:   17
        % PTS CARD.:   60
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 9.5e-16
        
        xyw=[
            4.17103444361599184908584447839530e-01 4.17103444361599184908584447839530e-01 1.36554632640510497515462517981177e-02
            4.17103444361599184908584447839530e-01 1.65793111276801630182831104320940e-01 1.36554632640510497515462517981177e-02
            1.65793111276801630182831104320940e-01 4.17103444361599184908584447839530e-01 1.36554632640510497515462517981177e-02
            1.80358116266370549496400599309709e-01 1.80358116266370549496400599309709e-01 1.31563152940089890530206417906811e-02
            1.80358116266370549496400599309709e-01 6.39283767467258901007198801380582e-01 1.31563152940089890530206417906811e-02
            6.39283767467258901007198801380582e-01 1.80358116266370549496400599309709e-01 1.31563152940089890530206417906811e-02
            2.85706502436586573523413790098857e-01 2.85706502436586573523413790098857e-01 1.88581185763976380553863521072344e-02
            2.85706502436586573523413790098857e-01 4.28586995126826852953172419802286e-01 1.88581185763976380553863521072344e-02
            4.28586995126826852953172419802286e-01 2.85706502436586573523413790098857e-01 1.88581185763976380553863521072344e-02
            6.66540634795968062187299096876814e-02 6.66540634795968062187299096876814e-02 6.22950040115271934382246143968587e-03
            6.66540634795968062187299096876814e-02 8.66691873040806415318115796253551e-01 6.22950040115271934382246143968587e-03
            8.66691873040806415318115796253551e-01 6.66540634795968062187299096876814e-02 6.22950040115271934382246143968587e-03
            1.47554916607538222361739599364228e-02 1.47554916607538222361739599364228e-02 1.38694378881882088294652533022600e-03
            1.47554916607538222361739599364228e-02 9.70489016678492300016500848869327e-01 1.38694378881882088294652533022600e-03
            9.70489016678492300016500848869327e-01 1.47554916607538222361739599364228e-02 1.38694378881882088294652533022600e-03
            4.65597871618890268852197777960100e-01 4.65597871618890268852197777960100e-01 1.25097254752486748002882066543862e-02
            4.65597871618890268852197777960100e-01 6.88042567622194622956044440797996e-02 1.25097254752486748002882066543862e-02
            6.88042567622194622956044440797996e-02 4.65597871618890268852197777960100e-01 1.25097254752486748002882066543862e-02
            7.25054707990023805663781786279287e-02 1.15751759031806882721538087821500e-02 2.29217420086793308500738852728773e-03
            7.25054707990023805663781786279287e-02 9.15919353297816929426744536613114e-01 2.29217420086793308500738852728773e-03
            1.15751759031806882721538087821500e-02 7.25054707990023805663781786279287e-02 2.29217420086793308500738852728773e-03
            1.15751759031806882721538087821500e-02 9.15919353297816929426744536613114e-01 2.29217420086793308500738852728773e-03
            9.15919353297816929426744536613114e-01 7.25054707990023805663781786279287e-02 2.29217420086793308500738852728773e-03
            9.15919353297816929426744536613114e-01 1.15751759031806882721538087821500e-02 2.29217420086793308500738852728773e-03
            4.15475459295229054834663884321344e-01 1.32296727600869565288910223443963e-02 5.19921997791976744918152064656169e-03
            4.15475459295229054834663884321344e-01 5.71294867944684092719853651942685e-01 5.19921997791976744918152064656169e-03
            1.32296727600869565288910223443963e-02 4.15475459295229054834663884321344e-01 5.19921997791976744918152064656169e-03
            1.32296727600869565288910223443963e-02 5.71294867944684092719853651942685e-01 5.19921997791976744918152064656169e-03
            5.71294867944684092719853651942685e-01 4.15475459295229054834663884321344e-01 5.19921997791976744918152064656169e-03
            5.71294867944684092719853651942685e-01 1.32296727600869565288910223443963e-02 5.19921997791976744918152064656169e-03
            2.71791870055354767288235962041654e-01 1.31358708340027583588716098006444e-02 4.34610725050059432117421920338529e-03
            2.71791870055354767288235962041654e-01 7.15072259110642427515358576783910e-01 4.34610725050059432117421920338529e-03
            1.31358708340027583588716098006444e-02 2.71791870055354767288235962041654e-01 4.34610725050059432117421920338529e-03
            1.31358708340027583588716098006444e-02 7.15072259110642427515358576783910e-01 4.34610725050059432117421920338529e-03
            7.15072259110642427515358576783910e-01 2.71791870055354767288235962041654e-01 4.34610725050059432117421920338529e-03
            7.15072259110642427515358576783910e-01 1.31358708340027583588716098006444e-02 4.34610725050059432117421920338529e-03
            2.99218942476970284349846451732446e-01 1.57505477926869974503176763391821e-01 1.30858129676684892261917525502213e-02
            2.99218942476970284349846451732446e-01 5.43275579596159796658128016133560e-01 1.30858129676684892261917525502213e-02
            1.57505477926869974503176763391821e-01 2.99218942476970284349846451732446e-01 1.30858129676684892261917525502213e-02
            1.57505477926869974503176763391821e-01 5.43275579596159796658128016133560e-01 1.30858129676684892261917525502213e-02
            5.43275579596159796658128016133560e-01 2.99218942476970284349846451732446e-01 1.30858129676684892261917525502213e-02
            5.43275579596159796658128016133560e-01 1.57505477926869974503176763391821e-01 1.30858129676684892261917525502213e-02
            3.06281591746186521163508587051183e-01 6.73493778673612542018389603981632e-02 1.12438862733455317843844767367045e-02
            3.06281591746186521163508587051183e-01 6.26369030386452196879076836921740e-01 1.12438862733455317843844767367045e-02
            6.73493778673612542018389603981632e-02 3.06281591746186521163508587051183e-01 1.12438862733455317843844767367045e-02
            6.73493778673612542018389603981632e-02 6.26369030386452196879076836921740e-01 1.12438862733455317843844767367045e-02
            6.26369030386452196879076836921740e-01 3.06281591746186521163508587051183e-01 1.12438862733455317843844767367045e-02
            6.26369030386452196879076836921740e-01 6.73493778673612542018389603981632e-02 1.12438862733455317843844767367045e-02
            1.68722513495259435201134579074278e-01 7.80423405682824783413664704312396e-02 1.02789491602272558407049629636276e-02
            1.68722513495259435201134579074278e-01 7.53235145936458017068559911422199e-01 1.02789491602272558407049629636276e-02
            7.80423405682824783413664704312396e-02 1.68722513495259435201134579074278e-01 1.02789491602272558407049629636276e-02
            7.80423405682824783413664704312396e-02 7.53235145936458017068559911422199e-01 1.02789491602272558407049629636276e-02
            7.53235145936458017068559911422199e-01 1.68722513495259435201134579074278e-01 1.02789491602272558407049629636276e-02
            7.53235145936458017068559911422199e-01 7.80423405682824783413664704312396e-02 1.02789491602272558407049629636276e-02
            1.59192287472792681768396505503915e-01 1.60176423621193435298248886056172e-02 3.98915010296479587842410907683188e-03
            1.59192287472792681768396505503915e-01 8.24790070165087985110119461751310e-01 3.98915010296479587842410907683188e-03
            1.60176423621193435298248886056172e-02 1.59192287472792681768396505503915e-01 3.98915010296479587842410907683188e-03
            1.60176423621193435298248886056172e-02 8.24790070165087985110119461751310e-01 3.98915010296479587842410907683188e-03
            8.24790070165087985110119461751310e-01 1.59192287472792681768396505503915e-01 3.98915010296479587842410907683188e-03
            8.24790070165087985110119461751310e-01 1.60176423621193435298248886056172e-02 3.98915010296479587842410907683188e-03
            ];
        
    case  18
        
        % ALG. DEG.:   18
        % PTS CARD.:   67
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.5e-15
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 1.53742606195579230299319561936500e-02
            4.74918211324045713705999105513911e-01 4.74918211324045713705999105513911e-01 6.55351374586937620442261831499309e-03
            4.74918211324045713705999105513911e-01 5.01635773519085725880017889721785e-02 6.55351374586937620442261831499309e-03
            5.01635773519085725880017889721785e-02 4.74918211324045713705999105513911e-01 6.55351374586937620442261831499309e-03
            1.51638506972604725220321597589646e-01 1.51638506972604725220321597589646e-01 1.01591694227291962776549283375971e-02
            1.51638506972604725220321597589646e-01 6.96722986054790549559356804820709e-01 1.01591694227291962776549283375971e-02
            6.96722986054790549559356804820709e-01 1.51638506972604725220321597589646e-01 1.01591694227291962776549283375971e-02
            4.11067101875919438391804305865662e-01 4.11067101875919438391804305865662e-01 1.67359970299239407820479641486600e-02
            4.11067101875919438391804305865662e-01 1.77865796248161123216391388268676e-01 1.67359970299239407820479641486600e-02
            1.77865796248161123216391388268676e-01 4.11067101875919438391804305865662e-01 1.67359970299239407820479641486600e-02
            2.65614609905374166043401373826782e-01 2.65614609905374166043401373826782e-01 1.55581983010030597963657683635574e-02
            2.65614609905374166043401373826782e-01 4.68770780189251667913197252346436e-01 1.55581983010030597963657683635574e-02
            4.68770780189251667913197252346436e-01 2.65614609905374166043401373826782e-01 1.55581983010030597963657683635574e-02
            3.75894434106820423541206466211406e-03 3.75894434106820423541206466211406e-03 2.66002808473890202996953169289895e-04
            3.75894434106820423541206466211406e-03 9.92482111317863568977770682977280e-01 2.66002808473890202996953169289895e-04
            9.92482111317863568977770682977280e-01 3.75894434106820423541206466211406e-03 2.66002808473890202996953169289895e-04
            7.24387055673327562299590454131248e-02 7.24387055673327562299590454131248e-02 6.89514330238346750262534712305751e-03
            7.24387055673327562299590454131248e-02 8.55122588865334432028930677915923e-01 6.89514330238346750262534712305751e-03
            8.55122588865334432028930677915923e-01 7.24387055673327562299590454131248e-02 6.89514330238346750262534712305751e-03
            3.85044034413163649421818490736769e-01 9.04270403543406542379301527034841e-02 7.66412909727656743719137466541724e-03
            3.85044034413163649421818490736769e-01 5.24528925232495724095826972188661e-01 7.66412909727656743719137466541724e-03
            9.04270403543406542379301527034841e-02 3.85044034413163649421818490736769e-01 7.66412909727656743719137466541724e-03
            9.04270403543406542379301527034841e-02 5.24528925232495724095826972188661e-01 7.66412909727656743719137466541724e-03
            5.24528925232495724095826972188661e-01 3.85044034413163649421818490736769e-01 7.66412909727656743719137466541724e-03
            5.24528925232495724095826972188661e-01 9.04270403543406542379301527034841e-02 7.66412909727656743719137466541724e-03
            4.72761418326517468546654754391056e-02 1.24989324834954822901478266317099e-02 2.10875838737222074531496573968070e-03
            4.72761418326517468546654754391056e-02 9.40224925683852852387190068839118e-01 2.10875838737222074531496573968070e-03
            1.24989324834954822901478266317099e-02 4.72761418326517468546654754391056e-02 2.10875838737222074531496573968070e-03
            1.24989324834954822901478266317099e-02 9.40224925683852852387190068839118e-01 2.10875838737222074531496573968070e-03
            9.40224925683852852387190068839118e-01 4.72761418326517468546654754391056e-02 2.10875838737222074531496573968070e-03
            9.40224925683852852387190068839118e-01 1.24989324834954822901478266317099e-02 2.10875838737222074531496573968070e-03
            3.02061957712870754466649714231608e-01 5.40117353390242999067183404804382e-02 8.18295420699327938351963496188546e-03
            3.02061957712870754466649714231608e-01 6.43926306948105042771146599989152e-01 8.18295420699327938351963496188546e-03
            5.40117353390242999067183404804382e-02 3.02061957712870754466649714231608e-01 8.18295420699327938351963496188546e-03
            5.40117353390242999067183404804382e-02 6.43926306948105042771146599989152e-01 8.18295420699327938351963496188546e-03
            6.43926306948105042771146599989152e-01 3.02061957712870754466649714231608e-01 8.18295420699327938351963496188546e-03
            6.43926306948105042771146599989152e-01 5.40117353390242999067183404804382e-02 8.18295420699327938351963496188546e-03
            2.56506159774241515592052564898040e-01 1.05050188192419655802689604229272e-02 3.86491764000311240415119762303675e-03
            2.56506159774241515592052564898040e-01 7.32988821406516555256871470191982e-01 3.86491764000311240415119762303675e-03
            1.05050188192419655802689604229272e-02 2.56506159774241515592052564898040e-01 3.86491764000311240415119762303675e-03
            1.05050188192419655802689604229272e-02 7.32988821406516555256871470191982e-01 3.86491764000311240415119762303675e-03
            7.32988821406516555256871470191982e-01 2.56506159774241515592052564898040e-01 3.86491764000311240415119762303675e-03
            7.32988821406516555256871470191982e-01 1.05050188192419655802689604229272e-02 3.86491764000311240415119762303675e-03
            1.78479125565887630777694994321791e-01 6.61224580284034413324434353853576e-02 8.45582695874003489877601680291264e-03
            1.78479125565887630777694994321791e-01 7.55398416405708927889861570292851e-01 8.45582695874003489877601680291264e-03
            6.61224580284034413324434353853576e-02 1.78479125565887630777694994321791e-01 8.45582695874003489877601680291264e-03
            6.61224580284034413324434353853576e-02 7.55398416405708927889861570292851e-01 8.45582695874003489877601680291264e-03
            7.55398416405708927889861570292851e-01 1.78479125565887630777694994321791e-01 8.45582695874003489877601680291264e-03
            7.55398416405708927889861570292851e-01 6.61224580284034413324434353853576e-02 8.45582695874003489877601680291264e-03
            2.68573306396013788965859703239403e-01 1.49066910125773915973823591230030e-01 1.37964432442897327879549607132503e-02
            2.68573306396013788965859703239403e-01 5.82359783478212267304741089901654e-01 1.37964432442897327879549607132503e-02
            1.49066910125773915973823591230030e-01 2.68573306396013788965859703239403e-01 1.37964432442897327879549607132503e-02
            1.49066910125773915973823591230030e-01 5.82359783478212267304741089901654e-01 1.37964432442897327879549607132503e-02
            5.82359783478212267304741089901654e-01 2.68573306396013788965859703239403e-01 1.37964432442897327879549607132503e-02
            5.82359783478212267304741089901654e-01 1.49066910125773915973823591230030e-01 1.37964432442897327879549607132503e-02
            4.11065668674618356348560155311134e-01 1.16918246746671606467460335920805e-02 4.79306223718075057743659428410865e-03
            4.11065668674618356348560155311134e-01 5.77242506650714526372780710516963e-01 4.79306223718075057743659428410865e-03
            1.16918246746671606467460335920805e-02 4.11065668674618356348560155311134e-01 4.79306223718075057743659428410865e-03
            1.16918246746671606467460335920805e-02 5.77242506650714526372780710516963e-01 4.79306223718075057743659428410865e-03
            5.77242506650714526372780710516963e-01 4.11065668674618356348560155311134e-01 4.79306223718075057743659428410865e-03
            5.77242506650714526372780710516963e-01 1.16918246746671606467460335920805e-02 4.79306223718075057743659428410865e-03
            1.32778830271389286510341776192945e-01 1.43315247789419925039666736665822e-02 3.82085248635981658549276573921816e-03
            1.32778830271389286510341776192945e-01 8.52889644949668790374630589212757e-01 3.82085248635981658549276573921816e-03
            1.43315247789419925039666736665822e-02 1.32778830271389286510341776192945e-01 3.82085248635981658549276573921816e-03
            1.43315247789419925039666736665822e-02 8.52889644949668790374630589212757e-01 3.82085248635981658549276573921816e-03
            8.52889644949668790374630589212757e-01 1.32778830271389286510341776192945e-01 3.82085248635981658549276573921816e-03
            8.52889644949668790374630589212757e-01 1.43315247789419925039666736665822e-02 3.82085248635981658549276573921816e-03
            ];
        
    case  19
        
        % ALG. DEG.:   19
        % PTS CARD.:   73
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.1e-15
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 1.72345804254526238641354041192244e-02
            5.25262798541034220400902654546371e-02 5.25262798541034220400902654546371e-02 3.55469681139747089790303924417003e-03
            5.25262798541034220400902654546371e-02 8.94947440291793183675395084719639e-01 3.55469681139747089790303924417003e-03
            8.94947440291793183675395084719639e-01 5.25262798541034220400902654546371e-02 3.55469681139747089790303924417003e-03
            1.11448055716998548070328922676708e-01 1.11448055716998548070328922676708e-01 7.61747825850241262746198600552816e-03
            1.11448055716998548070328922676708e-01 7.77103888566002876103766539017670e-01 7.61747825850241262746198600552816e-03
            7.77103888566002876103766539017670e-01 1.11448055716998548070328922676708e-01 7.61747825850241262746198600552816e-03
            1.16390273279224441133061773712143e-02 1.16390273279224441133061773712143e-02 8.82596209154269347764432129821444e-04
            1.16390273279224441133061773712143e-02 9.76721945344155129120622405025642e-01 8.82596209154269347764432129821444e-04
            9.76721945344155129120622405025642e-01 1.16390273279224441133061773712143e-02 8.82596209154269347764432129821444e-04
            2.55162133153124748297102541982895e-01 2.55162133153124748297102541982895e-01 1.58764272937649796035852745035299e-02
            2.55162133153124748297102541982895e-01 4.89675733693750503405794916034210e-01 1.58764272937649796035852745035299e-02
            4.89675733693750503405794916034210e-01 2.55162133153124748297102541982895e-01 1.58764272937649796035852745035299e-02
            4.03969717966386032692582830350148e-01 4.03969717966386032692582830350148e-01 1.57686793226197961848011885876986e-02
            4.03969717966386032692582830350148e-01 1.92060564067227934614834339299705e-01 1.57686793226197961848011885876986e-02
            1.92060564067227934614834339299705e-01 4.03969717966386032692582830350148e-01 1.57686793226197961848011885876986e-02
            1.78171006079627408658083709269704e-01 1.78171006079627408658083709269704e-01 1.23259905267924067817641997635292e-02
            1.78171006079627408658083709269704e-01 6.43657987840745127172681350202765e-01 1.23259905267924067817641997635292e-02
            6.43657987840745127172681350202765e-01 1.78171006079627408658083709269704e-01 1.23259905267924067817641997635292e-02
            4.59194388956827626024903565848945e-01 4.59194388956827626024903565848945e-01 1.14917854885616173982176846379843e-02
            4.59194388956827626024903565848945e-01 8.16112220863447479501928683021106e-02 1.14917854885616173982176846379843e-02
            8.16112220863447479501928683021106e-02 4.59194388956827626024903565848945e-01 1.14917854885616173982176846379843e-02
            4.92512449865874224474993070543860e-01 4.92512449865874224474993070543860e-01 5.16094109120942678287091709421475e-03
            4.92512449865874224474993070543860e-01 1.49751002682515510500138589122798e-02 5.16094109120942678287091709421475e-03
            1.49751002682515510500138589122798e-02 4.92512449865874224474993070543860e-01 5.16094109120942678287091709421475e-03
            1.42422282571126906924874333526532e-01 5.00514235235043503796648067805108e-03 1.46284624394003468991765082307666e-03
            1.42422282571126906924874333526532e-01 8.52572575076522731762906914809719e-01 1.46284624394003468991765082307666e-03
            5.00514235235043503796648067805108e-03 1.42422282571126906924874333526532e-01 1.46284624394003468991765082307666e-03
            5.00514235235043503796648067805108e-03 8.52572575076522731762906914809719e-01 1.46284624394003468991765082307666e-03
            8.52572575076522731762906914809719e-01 1.42422282571126906924874333526532e-01 1.46284624394003468991765082307666e-03
            8.52572575076522731762906914809719e-01 5.00514235235043503796648067805108e-03 1.46284624394003468991765082307666e-03
            6.00838999627023515048840351937542e-02 9.77706143867685707748993451104980e-03 1.66369442029695138902944950842766e-03
            6.00838999627023515048840351937542e-02 9.30139038598620793152349506272003e-01 1.66369442029695138902944950842766e-03
            9.77706143867685707748993451104980e-03 6.00838999627023515048840351937542e-02 1.66369442029695138902944950842766e-03
            9.77706143867685707748993451104980e-03 9.30139038598620793152349506272003e-01 1.66369442029695138902944950842766e-03
            9.30139038598620793152349506272003e-01 6.00838999627023515048840351937542e-02 1.66369442029695138902944950842766e-03
            9.30139038598620793152349506272003e-01 9.77706143867685707748993451104980e-03 1.66369442029695138902944950842766e-03
            1.30700669960534532343388036679244e-01 3.91424494346088655549742441053240e-02 4.84775954081209745766267715794129e-03
            1.30700669960534532343388036679244e-01 8.30156880604856595162743815308204e-01 4.84775954081209745766267715794129e-03
            3.91424494346088655549742441053240e-02 1.30700669960534532343388036679244e-01 4.84775954081209745766267715794129e-03
            3.91424494346088655549742441053240e-02 8.30156880604856595162743815308204e-01 4.84775954081209745766267715794129e-03
            8.30156880604856595162743815308204e-01 1.30700669960534532343388036679244e-01 4.84775954081209745766267715794129e-03
            8.30156880604856595162743815308204e-01 3.91424494346088655549742441053240e-02 4.84775954081209745766267715794129e-03
            3.11318383223986860919296759675490e-01 1.29312809767979042252505905707949e-01 1.31731323537226732711680199372495e-02
            3.11318383223986860919296759675490e-01 5.59368807008034041317046103358734e-01 1.31731323537226732711680199372495e-02
            1.29312809767979042252505905707949e-01 3.11318383223986860919296759675490e-01 1.31731323537226732711680199372495e-02
            1.29312809767979042252505905707949e-01 5.59368807008034041317046103358734e-01 1.31731323537226732711680199372495e-02
            5.59368807008034041317046103358734e-01 3.11318383223986860919296759675490e-01 1.31731323537226732711680199372495e-02
            5.59368807008034041317046103358734e-01 1.29312809767979042252505905707949e-01 1.31731323537226732711680199372495e-02
            2.21433941889113417023082774903742e-01 7.45611893043551660076317944003677e-02 9.05403729521524550105660011922737e-03
            2.21433941889113417023082774903742e-01 7.04004868806531458602648854139261e-01 9.05403729521524550105660011922737e-03
            7.45611893043551660076317944003677e-02 2.21433941889113417023082774903742e-01 9.05403729521524550105660011922737e-03
            7.45611893043551660076317944003677e-02 7.04004868806531458602648854139261e-01 9.05403729521524550105660011922737e-03
            7.04004868806531458602648854139261e-01 2.21433941889113417023082774903742e-01 9.05403729521524550105660011922737e-03
            7.04004868806531458602648854139261e-01 7.45611893043551660076317944003677e-02 9.05403729521524550105660011922737e-03
            3.54025926999711915144075646821875e-01 4.08883144649781413160205545409553e-02 8.05110473046970892552209875248082e-03
            3.54025926999711915144075646821875e-01 6.05085758535309992112161125987768e-01 8.05110473046970892552209875248082e-03
            4.08883144649781413160205545409553e-02 3.54025926999711915144075646821875e-01 8.05110473046970892552209875248082e-03
            4.08883144649781413160205545409553e-02 6.05085758535309992112161125987768e-01 8.05110473046970892552209875248082e-03
            6.05085758535309992112161125987768e-01 3.54025926999711915144075646821875e-01 8.05110473046970892552209875248082e-03
            6.05085758535309992112161125987768e-01 4.08883144649781413160205545409553e-02 8.05110473046970892552209875248082e-03
            2.41894104006892624036950678600988e-01 1.49236389074384866865186083373374e-02 4.22796241954673680113474532049622e-03
            2.41894104006892624036950678600988e-01 7.43182257085668873664019429270411e-01 4.22796241954673680113474532049622e-03
            1.49236389074384866865186083373374e-02 2.41894104006892624036950678600988e-01 4.22796241954673680113474532049622e-03
            1.49236389074384866865186083373374e-02 7.43182257085668873664019429270411e-01 4.22796241954673680113474532049622e-03
            7.43182257085668873664019429270411e-01 2.41894104006892624036950678600988e-01 4.22796241954673680113474532049622e-03
            7.43182257085668873664019429270411e-01 1.49236389074384866865186083373374e-02 4.22796241954673680113474532049622e-03
            3.64620414338710008284039076897898e-01 2.06910384910238874869858882732387e-03 1.64106875741986779453085443236660e-03
            3.64620414338710008284039076897898e-01 6.33310481812187631156518818897894e-01 1.64106875741986779453085443236660e-03
            2.06910384910238874869858882732387e-03 3.64620414338710008284039076897898e-01 1.64106875741986779453085443236660e-03
            2.06910384910238874869858882732387e-03 6.33310481812187631156518818897894e-01 1.64106875741986779453085443236660e-03
            6.33310481812187631156518818897894e-01 3.64620414338710008284039076897898e-01 1.64106875741986779453085443236660e-03
            6.33310481812187631156518818897894e-01 2.06910384910238874869858882732387e-03 1.64106875741986779453085443236660e-03
            ];
        
    case  20
        
        % ALG. DEG.:   20
        % PTS CARD.:   79
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.2e-15
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 1.39101107014531263222911761090472e-02
            1.86294997744540891115860858917586e-01 1.86294997744540891115860858917586e-01 9.17346297425292167559973677271046e-03
            1.86294997744540891115860858917586e-01 6.27410004510918217768278282164829e-01 9.17346297425292167559973677271046e-03
            6.27410004510918217768278282164829e-01 1.86294997744540891115860858917586e-01 9.17346297425292167559973677271046e-03
            3.73108805988845854129110080066312e-02 3.73108805988845854129110080066312e-02 2.16127541066557906160316981925007e-03
            3.73108805988845854129110080066312e-02 9.25378238802230801418602368357824e-01 2.16127541066557906160316981925007e-03
            9.25378238802230801418602368357824e-01 3.73108805988845854129110080066312e-02 2.16127541066557906160316981925007e-03
            4.76245611540498992031444913664018e-01 4.76245611540498992031444913664018e-01 7.10182530340844678229395015023329e-03
            4.76245611540498992031444913664018e-01 4.75087769190020159371101726719644e-02 7.10182530340844678229395015023329e-03
            4.75087769190020159371101726719644e-02 4.76245611540498992031444913664018e-01 7.10182530340844678229395015023329e-03
            4.45551056955924784652722792088753e-01 4.45551056955924784652722792088753e-01 9.45239993323245680789668199395237e-03
            4.45551056955924784652722792088753e-01 1.08897886088150430694554415822495e-01 9.45239993323245680789668199395237e-03
            1.08897886088150430694554415822495e-01 4.45551056955924784652722792088753e-01 9.45239993323245680789668199395237e-03
            2.54579267673339049160574631969212e-01 2.54579267673339049160574631969212e-01 1.40832013075202593099799486253687e-02
            2.54579267673339049160574631969212e-01 4.90841464653321901678850736061577e-01 1.40832013075202593099799486253687e-02
            4.90841464653321901678850736061577e-01 2.54579267673339049160574631969212e-01 1.40832013075202593099799486253687e-02
            3.93425347817099813063634883292252e-01 3.93425347817099813063634883292252e-01 1.37880506290704689387194292748973e-02
            3.93425347817099813063634883292252e-01 2.13149304365800373872730233415496e-01 1.37880506290704689387194292748973e-02
            2.13149304365800373872730233415496e-01 3.93425347817099813063634883292252e-01 1.37880506290704689387194292748973e-02
            1.09761410283976314911091876069804e-02 1.09761410283976314911091876069804e-02 7.98840791066620617595639419761255e-04
            1.09761410283976314911091876069804e-02 9.78047717943204775181698096275795e-01 7.98840791066620617595639419761255e-04
            9.78047717943204775181698096275795e-01 1.09761410283976314911091876069804e-02 7.98840791066620617595639419761255e-04
            1.09383596711714464744247266025923e-01 1.09383596711714464744247266025923e-01 7.83023077607454022486255240664832e-03
            1.09383596711714464744247266025923e-01 7.81232806576571015000354236690328e-01 7.83023077607454022486255240664832e-03
            7.81232806576571015000354236690328e-01 1.09383596711714464744247266025923e-01 7.83023077607454022486255240664832e-03
            6.40905856084340308509439410045161e-02 4.85493760762382986495566683515790e-03 1.12986960212586645332932810958937e-03
            6.40905856084340308509439410045161e-02 9.31054476783942153161888199974783e-01 1.12986960212586645332932810958937e-03
            4.85493760762382986495566683515790e-03 6.40905856084340308509439410045161e-02 1.12986960212586645332932810958937e-03
            4.85493760762382986495566683515790e-03 9.31054476783942153161888199974783e-01 1.12986960212586645332932810958937e-03
            9.31054476783942153161888199974783e-01 6.40905856084340308509439410045161e-02 1.12986960212586645332932810958937e-03
            9.31054476783942153161888199974783e-01 4.85493760762382986495566683515790e-03 1.12986960212586645332932810958937e-03
            2.15607057390094364324184539327689e-01 1.06227204720270099191914425773575e-01 7.72260782209923703212073320401032e-03
            2.15607057390094364324184539327689e-01 6.78165737889635522606113227084279e-01 7.72260782209923703212073320401032e-03
            1.06227204720270099191914425773575e-01 2.15607057390094364324184539327689e-01 7.72260782209923703212073320401032e-03
            1.06227204720270099191914425773575e-01 6.78165737889635522606113227084279e-01 7.72260782209923703212073320401032e-03
            6.78165737889635522606113227084279e-01 2.15607057390094364324184539327689e-01 7.72260782209923703212073320401032e-03
            6.78165737889635522606113227084279e-01 1.06227204720270099191914425773575e-01 7.72260782209923703212073320401032e-03
            1.59133707657067219320978779251163e-01 7.57078050469658144633333662909536e-03 2.20289741855849959331625775860175e-03
            1.59133707657067219320978779251163e-01 8.33295511838236135915281010966282e-01 2.20289741855849959331625775860175e-03
            7.57078050469658144633333662909536e-03 1.59133707657067219320978779251163e-01 2.20289741855849959331625775860175e-03
            7.57078050469658144633333662909536e-03 8.33295511838236135915281010966282e-01 2.20289741855849959331625775860175e-03
            8.33295511838236135915281010966282e-01 1.59133707657067219320978779251163e-01 2.20289741855849959331625775860175e-03
            8.33295511838236135915281010966282e-01 7.57078050469658144633333662909536e-03 2.20289741855849959331625775860175e-03
            3.17860123835771946065875681597390e-01 1.39808071991799981459436708064459e-01 1.16917457318277458883093444796941e-02
            3.17860123835771946065875681597390e-01 5.42331804172428100230263225967064e-01 1.16917457318277458883093444796941e-02
            1.39808071991799981459436708064459e-01 3.17860123835771946065875681597390e-01 1.16917457318277458883093444796941e-02
            1.39808071991799981459436708064459e-01 5.42331804172428100230263225967064e-01 1.16917457318277458883093444796941e-02
            5.42331804172428100230263225967064e-01 3.17860123835771946065875681597390e-01 1.16917457318277458883093444796941e-02
            5.42331804172428100230263225967064e-01 1.39808071991799981459436708064459e-01 1.16917457318277458883093444796941e-02
            1.98518132228788168891497889489983e-01 4.65603649076643533177843892190140e-02 5.98639857895469536253063580488742e-03
            1.98518132228788168891497889489983e-01 7.54921502863547533301868952548830e-01 5.98639857895469536253063580488742e-03
            4.65603649076643533177843892190140e-02 1.98518132228788168891497889489983e-01 5.98639857895469536253063580488742e-03
            4.65603649076643533177843892190140e-02 7.54921502863547533301868952548830e-01 5.98639857895469536253063580488742e-03
            7.54921502863547533301868952548830e-01 1.98518132228788168891497889489983e-01 5.98639857895469536253063580488742e-03
            7.54921502863547533301868952548830e-01 4.65603649076643533177843892190140e-02 5.98639857895469536253063580488742e-03
            9.99522962881386201061673091317061e-02 3.83636847753746762501592115768290e-02 4.14571152761386129553811130676877e-03
            9.99522962881386201061673091317061e-02 8.61684018936486717521461287105922e-01 4.14571152761386129553811130676877e-03
            3.83636847753746762501592115768290e-02 9.99522962881386201061673091317061e-02 4.14571152761386129553811130676877e-03
            3.83636847753746762501592115768290e-02 8.61684018936486717521461287105922e-01 4.14571152761386129553811130676877e-03
            8.61684018936486717521461287105922e-01 9.99522962881386201061673091317061e-02 4.14571152761386129553811130676877e-03
            8.61684018936486717521461287105922e-01 3.83636847753746762501592115768290e-02 4.14571152761386129553811130676877e-03
            4.20023758816224057621724341515801e-01 9.83154829280259191792712414326161e-03 3.69568150025530129873518880856409e-03
            4.20023758816224057621724341515801e-01 5.70144692890973248111663451709319e-01 3.69568150025530129873518880856409e-03
            9.83154829280259191792712414326161e-03 4.20023758816224057621724341515801e-01 3.69568150025530129873518880856409e-03
            9.83154829280259191792712414326161e-03 5.70144692890973248111663451709319e-01 3.69568150025530129873518880856409e-03
            5.70144692890973248111663451709319e-01 4.20023758816224057621724341515801e-01 3.69568150025530129873518880856409e-03
            5.70144692890973248111663451709319e-01 9.83154829280259191792712414326161e-03 3.69568150025530129873518880856409e-03
            3.33134817309587438760587474462227e-01 5.49874791429868711722939167430013e-02 8.66722556721933982959438225179838e-03
            3.33134817309587438760587474462227e-01 6.11877703547425766394951551774284e-01 8.66722556721933982959438225179838e-03
            5.49874791429868711722939167430013e-02 3.33134817309587438760587474462227e-01 8.66722556721933982959438225179838e-03
            5.49874791429868711722939167430013e-02 6.11877703547425766394951551774284e-01 8.66722556721933982959438225179838e-03
            6.11877703547425766394951551774284e-01 3.33134817309587438760587474462227e-01 8.66722556721933982959438225179838e-03
            6.11877703547425766394951551774284e-01 5.49874791429868711722939167430013e-02 8.66722556721933982959438225179838e-03
            2.80581411423665216808842615137110e-01 1.07372128560111156081458005928653e-02 3.57820023845768818773760422402574e-03
            2.80581411423665216808842615137110e-01 7.08681375720323636357989016687497e-01 3.57820023845768818773760422402574e-03
            1.07372128560111156081458005928653e-02 2.80581411423665216808842615137110e-01 3.57820023845768818773760422402574e-03
            1.07372128560111156081458005928653e-02 7.08681375720323636357989016687497e-01 3.57820023845768818773760422402574e-03
            7.08681375720323636357989016687497e-01 2.80581411423665216808842615137110e-01 3.57820023845768818773760422402574e-03
            7.08681375720323636357989016687497e-01 1.07372128560111156081458005928653e-02 3.57820023845768818773760422402574e-03
            ];
        
    case  21
        
        % ALG. DEG.:   21
        % PTS CARD.:   87
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.4e-15
        
        xyw=[
            2.98936235314982479138734561274759e-01 2.98936235314982479138734561274759e-01 1.07255609645661720286113904876402e-02
            2.98936235314982479138734561274759e-01 4.02127529370035041722530877450481e-01 1.07255609645661720286113904876402e-02
            4.02127529370035041722530877450481e-01 2.98936235314982479138734561274759e-01 1.07255609645661720286113904876402e-02
            4.97007875468685578201899488703930e-01 4.97007875468685578201899488703930e-01 2.21891484853293987494771144497463e-03
            4.97007875468685578201899488703930e-01 5.98424906262884359620102259214036e-03 2.21891484853293987494771144497463e-03
            5.98424906262884359620102259214036e-03 4.97007875468685578201899488703930e-01 2.21891484853293987494771144497463e-03
            4.03617586546385076484710907607223e-01 4.03617586546385076484710907607223e-01 1.15003523266419341308797186229640e-02
            4.03617586546385076484710907607223e-01 1.92764826907229847030578184785554e-01 1.15003523266419341308797186229640e-02
            1.92764826907229847030578184785554e-01 4.03617586546385076484710907607223e-01 1.15003523266419341308797186229640e-02
            1.18988577622719290949326875761471e-01 1.18988577622719290949326875761471e-01 6.82801622611509985427691304948894e-03
            1.18988577622719290949326875761471e-01 7.62022844754561390345770632848144e-01 6.82801622611509985427691304948894e-03
            7.62022844754561390345770632848144e-01 1.18988577622719290949326875761471e-01 6.82801622611509985427691304948894e-03
            1.90288718091278424848766803734179e-01 1.90288718091278424848766803734179e-01 9.72762093037535767459100810583550e-03
            1.90288718091278424848766803734179e-01 6.19422563817443094791315161273815e-01 9.72762093037535767459100810583550e-03
            6.19422563817443094791315161273815e-01 1.90288718091278424848766803734179e-01 9.72762093037535767459100810583550e-03
            4.81597868653216576451114860901725e-01 4.81597868653216576451114860901725e-01 6.10720508169219217231571406045987e-03
            4.81597868653216576451114860901725e-01 3.68042626935668470977702781965490e-02 6.10720508169219217231571406045987e-03
            3.68042626935668470977702781965490e-02 4.81597868653216576451114860901725e-01 6.10720508169219217231571406045987e-03
            4.49812791775362375013003202184336e-01 4.49812791775362375013003202184336e-01 9.80723761391201140358298005139659e-03
            4.49812791775362375013003202184336e-01 1.00374416449275249973993595631327e-01 9.80723761391201140358298005139659e-03
            1.00374416449275249973993595631327e-01 4.49812791775362375013003202184336e-01 9.80723761391201140358298005139659e-03
            5.36275755461448552785519439112250e-02 5.36275755461448552785519439112250e-02 3.57604255064182660270311053807291e-03
            5.36275755461448552785519439112250e-02 8.92744848907710331076259535620920e-01 3.57604255064182660270311053807291e-03
            8.92744848907710331076259535620920e-01 5.36275755461448552785519439112250e-02 3.57604255064182660270311053807291e-03
            1.07424564328282810821146853186292e-02 1.07424564328282810821146853186292e-02 7.54349636189344664553846353527433e-04
            1.07424564328282810821146853186292e-02 9.78515087134343430896876725455513e-01 7.54349636189344664553846353527433e-04
            9.78515087134343430896876725455513e-01 1.07424564328282810821146853186292e-02 7.54349636189344664553846353527433e-04
            2.89189496078594676653494843776571e-01 2.05295559335161609260111958974448e-01 8.74770807788156391782141696467079e-03
            2.89189496078594676653494843776571e-01 5.05514944586243686330817581620067e-01 8.74770807788156391782141696467079e-03
            2.05295559335161609260111958974448e-01 2.89189496078594676653494843776571e-01 8.74770807788156391782141696467079e-03
            2.05295559335161609260111958974448e-01 5.05514944586243686330817581620067e-01 8.74770807788156391782141696467079e-03
            5.05514944586243686330817581620067e-01 2.89189496078594676653494843776571e-01 8.74770807788156391782141696467079e-03
            5.05514944586243686330817581620067e-01 2.05295559335161609260111958974448e-01 8.74770807788156391782141696467079e-03
            2.37873382597993981990569523077284e-01 6.93180903146811862314269347962181e-03 2.10306014407486479775299947903022e-03
            2.37873382597993981990569523077284e-01 7.55194808370537873365435643790988e-01 2.10306014407486479775299947903022e-03
            6.93180903146811862314269347962181e-03 2.37873382597993981990569523077284e-01 2.10306014407486479775299947903022e-03
            6.93180903146811862314269347962181e-03 7.55194808370537873365435643790988e-01 2.10306014407486479775299947903022e-03
            7.55194808370537873365435643790988e-01 2.37873382597993981990569523077284e-01 2.10306014407486479775299947903022e-03
            7.55194808370537873365435643790988e-01 6.93180903146811862314269347962181e-03 2.10306014407486479775299947903022e-03
            3.18865310794828216156560074523441e-01 1.23779400405492817638375413480389e-01 9.22374242396641931540646197618116e-03
            3.18865310794828216156560074523441e-01 5.57355288799678882938337665109430e-01 9.22374242396641931540646197618116e-03
            1.23779400405492817638375413480389e-01 3.18865310794828216156560074523441e-01 9.22374242396641931540646197618116e-03
            1.23779400405492817638375413480389e-01 5.57355288799678882938337665109430e-01 9.22374242396641931540646197618116e-03
            5.57355288799678882938337665109430e-01 3.18865310794828216156560074523441e-01 9.22374242396641931540646197618116e-03
            5.57355288799678882938337665109430e-01 1.23779400405492817638375413480389e-01 9.22374242396641931540646197618116e-03
            2.31873625370400959777583693721681e-01 3.89913626232203450183710913279356e-02 5.23495209266242218781872708177616e-03
            2.31873625370400959777583693721681e-01 7.29135012006378646631787887599785e-01 5.23495209266242218781872708177616e-03
            3.89913626232203450183710913279356e-02 2.31873625370400959777583693721681e-01 5.23495209266242218781872708177616e-03
            3.89913626232203450183710913279356e-02 7.29135012006378646631787887599785e-01 5.23495209266242218781872708177616e-03
            7.29135012006378646631787887599785e-01 2.31873625370400959777583693721681e-01 5.23495209266242218781872708177616e-03
            7.29135012006378646631787887599785e-01 3.89913626232203450183710913279356e-02 5.23495209266242218781872708177616e-03
            1.33167122941370286159923352897749e-01 9.53624752971060293815774855374912e-03 2.24040656095073802481221214577545e-03
            1.33167122941370286159923352897749e-01 8.57296629528919162943623177852714e-01 2.24040656095073802481221214577545e-03
            9.53624752971060293815774855374912e-03 1.33167122941370286159923352897749e-01 2.24040656095073802481221214577545e-03
            9.53624752971060293815774855374912e-03 8.57296629528919162943623177852714e-01 2.24040656095073802481221214577545e-03
            8.57296629528919162943623177852714e-01 1.33167122941370286159923352897749e-01 2.24040656095073802481221214577545e-03
            8.57296629528919162943623177852714e-01 9.53624752971060293815774855374912e-03 2.24040656095073802481221214577545e-03
            3.46807979809911015589563021421782e-01 5.30521917012168400806793044921505e-02 7.25015295948551116711477959597687e-03
            3.46807979809911015589563021421782e-01 6.00139828488872040246349115477642e-01 7.25015295948551116711477959597687e-03
            5.30521917012168400806793044921505e-02 3.46807979809911015589563021421782e-01 7.25015295948551116711477959597687e-03
            5.30521917012168400806793044921505e-02 6.00139828488872040246349115477642e-01 7.25015295948551116711477959597687e-03
            6.00139828488872040246349115477642e-01 3.46807979809911015589563021421782e-01 7.25015295948551116711477959597687e-03
            6.00139828488872040246349115477642e-01 5.30521917012168400806793044921505e-02 7.25015295948551116711477959597687e-03
            2.16599623189982493087768489203881e-01 1.00458020074114501296591583923146e-01 7.95201835271398642379647725420000e-03
            2.16599623189982493087768489203881e-01 6.82942356735902977860064311244059e-01 7.95201835271398642379647725420000e-03
            1.00458020074114501296591583923146e-01 2.16599623189982493087768489203881e-01 7.95201835271398642379647725420000e-03
            1.00458020074114501296591583923146e-01 6.82942356735902977860064311244059e-01 7.95201835271398642379647725420000e-03
            6.82942356735902977860064311244059e-01 2.16599623189982493087768489203881e-01 7.95201835271398642379647725420000e-03
            6.82942356735902977860064311244059e-01 1.00458020074114501296591583923146e-01 7.95201835271398642379647725420000e-03
            1.28829807962051540171088959141343e-01 4.94510655685405711712299137161608e-02 4.90598591127520619603608054148935e-03
            1.28829807962051540171088959141343e-01 8.21719126469407812329848184162984e-01 4.90598591127520619603608054148935e-03
            4.94510655685405711712299137161608e-02 1.28829807962051540171088959141343e-01 4.90598591127520619603608054148935e-03
            4.94510655685405711712299137161608e-02 8.21719126469407812329848184162984e-01 4.90598591127520619603608054148935e-03
            8.21719126469407812329848184162984e-01 1.28829807962051540171088959141343e-01 4.90598591127520619603608054148935e-03
            8.21719126469407812329848184162984e-01 4.94510655685405711712299137161608e-02 4.90598591127520619603608054148935e-03
            3.60953408018922172750819754583063e-01 1.02546358729245206536928591845026e-02 3.41994242896715306551103097376654e-03
            3.60953408018922172750819754583063e-01 6.28791956108153304860763910255628e-01 3.41994242896715306551103097376654e-03
            1.02546358729245206536928591845026e-02 3.60953408018922172750819754583063e-01 3.41994242896715306551103097376654e-03
            1.02546358729245206536928591845026e-02 6.28791956108153304860763910255628e-01 3.41994242896715306551103097376654e-03
            6.28791956108153304860763910255628e-01 3.60953408018922172750819754583063e-01 3.41994242896715306551103097376654e-03
            6.28791956108153304860763910255628e-01 1.02546358729245206536928591845026e-02 3.41994242896715306551103097376654e-03
            5.57195650723719473895734211055242e-02 1.03019036434239090460618371025703e-02 1.63271429202204256848907082400046e-03
            5.57195650723719473895734211055242e-02 9.33978531284204160911599501559976e-01 1.63271429202204256848907082400046e-03
            1.03019036434239090460618371025703e-02 5.57195650723719473895734211055242e-02 1.63271429202204256848907082400046e-03
            1.03019036434239090460618371025703e-02 9.33978531284204160911599501559976e-01 1.63271429202204256848907082400046e-03
            9.33978531284204160911599501559976e-01 5.57195650723719473895734211055242e-02 1.63271429202204256848907082400046e-03
            9.33978531284204160911599501559976e-01 1.03019036434239090460618371025703e-02 1.63271429202204256848907082400046e-03
            ];
        
    case  22
        
        % ALG. DEG.:   22
        % PTS CARD.:   96
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.0e-15
        
        xyw=[
            3.85184524627302127264272257889388e-01 3.85184524627302127264272257889388e-01 6.74654194180532903318825432847916e-03
            3.85184524627302127264272257889388e-01 2.29630950745395745471455484221224e-01 6.74654194180532903318825432847916e-03
            2.29630950745395745471455484221224e-01 3.85184524627302127264272257889388e-01 6.74654194180532903318825432847916e-03
            4.57769411367672074408829985259217e-01 4.57769411367672074408829985259217e-01 6.93069976211709362634572073602612e-03
            4.57769411367672074408829985259217e-01 8.44611772646558511823400294815656e-02 6.93069976211709362634572073602612e-03
            8.44611772646558511823400294815656e-02 4.57769411367672074408829985259217e-01 6.93069976211709362634572073602612e-03
            2.94558259029950031404609944729600e-01 2.94558259029950031404609944729600e-01 1.05378819787260869650369343730745e-02
            2.94558259029950031404609944729600e-01 4.10883481940099937190780110540800e-01 1.05378819787260869650369343730745e-02
            4.10883481940099937190780110540800e-01 2.94558259029950031404609944729600e-01 1.05378819787260869650369343730745e-02
            1.88510523630283838292243103751389e-01 1.88510523630283838292243103751389e-01 8.01064956257444352183405555933859e-03
            1.88510523630283838292243103751389e-01 6.22978952739432267904362561239395e-01 8.01064956257444352183405555933859e-03
            6.22978952739432267904362561239395e-01 1.88510523630283838292243103751389e-01 8.01064956257444352183405555933859e-03
            4.21981888793534931636486362549476e-01 4.21981888793534931636486362549476e-01 9.42654627692064012167705300271336e-03
            4.21981888793534931636486362549476e-01 1.56036222412930136727027274901047e-01 9.42654627692064012167705300271336e-03
            1.56036222412930136727027274901047e-01 4.21981888793534931636486362549476e-01 9.42654627692064012167705300271336e-03
            4.96161178409708636394270797609352e-01 4.96161178409708636394270797609352e-01 2.64466983299220829636833052234124e-03
            4.96161178409708636394270797609352e-01 7.67764318058272721145840478129685e-03 2.64466983299220829636833052234124e-03
            7.67764318058272721145840478129685e-03 4.96161178409708636394270797609352e-01 2.64466983299220829636833052234124e-03
            2.91084706708073584491280882957653e-02 2.91084706708073584491280882957653e-02 1.78455458292818753586461699711663e-03
            2.91084706708073584491280882957653e-02 9.41783058658385296979531631222926e-01 1.78455458292818753586461699711663e-03
            9.41783058658385296979531631222926e-01 2.91084706708073584491280882957653e-02 1.78455458292818753586461699711663e-03
            1.15431538219204890549640651897789e-01 1.15431538219204890549640651897789e-01 7.20785656405229912396803726437611e-03
            1.15431538219204890549640651897789e-01 7.69136923561590246656294311833335e-01 7.20785656405229912396803726437611e-03
            7.69136923561590246656294311833335e-01 1.15431538219204890549640651897789e-01 7.20785656405229912396803726437611e-03
            6.98421694674436210892309873088379e-02 7.87628222158237752126286324028115e-03 1.29771923711563823451708987732900e-03
            6.98421694674436210892309873088379e-02 9.22281548310973975368654009798774e-01 1.29771923711563823451708987732900e-03
            7.87628222158237752126286324028115e-03 6.98421694674436210892309873088379e-02 1.29771923711563823451708987732900e-03
            7.87628222158237752126286324028115e-03 9.22281548310973975368654009798774e-01 1.29771923711563823451708987732900e-03
            9.22281548310973975368654009798774e-01 6.98421694674436210892309873088379e-02 1.29771923711563823451708987732900e-03
            9.22281548310973975368654009798774e-01 7.87628222158237752126286324028115e-03 1.29771923711563823451708987732900e-03
            9.03988311664077337503897524584318e-02 4.47522843483358911953473580069840e-02 3.75878890889418679557842928318223e-03
            9.03988311664077337503897524584318e-02 8.64848884485256430565414120792411e-01 3.75878890889418679557842928318223e-03
            4.47522843483358911953473580069840e-02 9.03988311664077337503897524584318e-02 3.75878890889418679557842928318223e-03
            4.47522843483358911953473580069840e-02 8.64848884485256430565414120792411e-01 3.75878890889418679557842928318223e-03
            8.64848884485256430565414120792411e-01 9.03988311664077337503897524584318e-02 3.75878890889418679557842928318223e-03
            8.64848884485256430565414120792411e-01 4.47522843483358911953473580069840e-02 3.75878890889418679557842928318223e-03
            4.11341764020558720016396136998083e-01 3.82752347008638374448175056841137e-02 5.59865673598138307437421801182609e-03
            4.11341764020558720016396136998083e-01 5.50383001278577421722104645596119e-01 5.59865673598138307437421801182609e-03
            3.82752347008638374448175056841137e-02 4.11341764020558720016396136998083e-01 5.59865673598138307437421801182609e-03
            3.82752347008638374448175056841137e-02 5.50383001278577421722104645596119e-01 5.59865673598138307437421801182609e-03
            5.50383001278577421722104645596119e-01 4.11341764020558720016396136998083e-01 5.59865673598138307437421801182609e-03
            5.50383001278577421722104645596119e-01 3.82752347008638374448175056841137e-02 5.59865673598138307437421801182609e-03
            3.32106105007446361643275167807587e-01 1.02747075986931432622917270691687e-01 8.85954674475510593756322208491838e-03
            3.32106105007446361643275167807587e-01 5.65146819005622247367170984944096e-01 8.85954674475510593756322208491838e-03
            1.02747075986931432622917270691687e-01 3.32106105007446361643275167807587e-01 8.85954674475510593756322208491838e-03
            1.02747075986931432622917270691687e-01 5.65146819005622247367170984944096e-01 8.85954674475510593756322208491838e-03
            5.65146819005622247367170984944096e-01 3.32106105007446361643275167807587e-01 8.85954674475510593756322208491838e-03
            5.65146819005622247367170984944096e-01 1.02747075986931432622917270691687e-01 8.85954674475510593756322208491838e-03
            3.62576280432467257242024061270058e-01 7.40024123471075441493827185013288e-03 2.45213019877848137015319274212288e-03
            3.62576280432467257242024061270058e-01 6.30023478332822017833336758485530e-01 2.45213019877848137015319274212288e-03
            7.40024123471075441493827185013288e-03 3.62576280432467257242024061270058e-01 2.45213019877848137015319274212288e-03
            7.40024123471075441493827185013288e-03 6.30023478332822017833336758485530e-01 2.45213019877848137015319274212288e-03
            6.30023478332822017833336758485530e-01 3.62576280432467257242024061270058e-01 2.45213019877848137015319274212288e-03
            6.30023478332822017833336758485530e-01 7.40024123471075441493827185013288e-03 2.45213019877848137015319274212288e-03
            2.90066824116668786626149767471361e-01 1.91081297966720159431375236636086e-01 1.08532097777544744854338887307676e-02
            2.90066824116668786626149767471361e-01 5.18851877916611026186899380263640e-01 1.08532097777544744854338887307676e-02
            1.91081297966720159431375236636086e-01 2.90066824116668786626149767471361e-01 1.08532097777544744854338887307676e-02
            1.91081297966720159431375236636086e-01 5.18851877916611026186899380263640e-01 1.08532097777544744854338887307676e-02
            5.18851877916611026186899380263640e-01 2.90066824116668786626149767471361e-01 1.08532097777544744854338887307676e-02
            5.18851877916611026186899380263640e-01 1.91081297966720159431375236636086e-01 1.08532097777544744854338887307676e-02
            2.87931802824171800825325817640987e-01 4.39916453934558610017191426777572e-02 5.83111143367149881849842429915043e-03
            2.87931802824171800825325817640987e-01 6.68076551782372352050742847495712e-01 5.83111143367149881849842429915043e-03
            4.39916453934558610017191426777572e-02 2.87931802824171800825325817640987e-01 5.83111143367149881849842429915043e-03
            4.39916453934558610017191426777572e-02 6.68076551782372352050742847495712e-01 5.83111143367149881849842429915043e-03
            6.68076551782372352050742847495712e-01 2.87931802824171800825325817640987e-01 5.83111143367149881849842429915043e-03
            6.68076551782372352050742847495712e-01 4.39916453934558610017191426777572e-02 5.83111143367149881849842429915043e-03
            2.16786933364941147850402103358647e-01 1.08689941862672029282421704010630e-01 7.85508131128515550700708303111242e-03
            2.16786933364941147850402103358647e-01 6.74523124772386850622751808259636e-01 7.85508131128515550700708303111242e-03
            1.08689941862672029282421704010630e-01 2.16786933364941147850402103358647e-01 7.85508131128515550700708303111242e-03
            1.08689941862672029282421704010630e-01 6.74523124772386850622751808259636e-01 7.85508131128515550700708303111242e-03
            6.74523124772386850622751808259636e-01 2.16786933364941147850402103358647e-01 7.85508131128515550700708303111242e-03
            6.74523124772386850622751808259636e-01 1.08689941862672029282421704010630e-01 7.85508131128515550700708303111242e-03
            1.45873719873525181656148674846918e-01 9.14471137496405787892239658276594e-03 2.05334353578777700083390911345305e-03
            1.45873719873525181656148674846918e-01 8.44981568751510758730205452593509e-01 2.05334353578777700083390911345305e-03
            9.14471137496405787892239658276594e-03 1.45873719873525181656148674846918e-01 2.05334353578777700083390911345305e-03
            9.14471137496405787892239658276594e-03 8.44981568751510758730205452593509e-01 2.05334353578777700083390911345305e-03
            8.44981568751510758730205452593509e-01 1.45873719873525181656148674846918e-01 2.05334353578777700083390911345305e-03
            8.44981568751510758730205452593509e-01 9.14471137496405787892239658276594e-03 2.05334353578777700083390911345305e-03
            1.76297434824500054428852990895393e-01 4.82549241146414051306656745055079e-02 5.28179248387344695003786654297073e-03
            1.76297434824500054428852990895393e-01 7.75447641060858505746011815062957e-01 5.28179248387344695003786654297073e-03
            4.82549241146414051306656745055079e-02 1.76297434824500054428852990895393e-01 5.28179248387344695003786654297073e-03
            4.82549241146414051306656745055079e-02 7.75447641060858505746011815062957e-01 5.28179248387344695003786654297073e-03
            7.75447641060858505746011815062957e-01 1.76297434824500054428852990895393e-01 5.28179248387344695003786654297073e-03
            7.75447641060858505746011815062957e-01 4.82549241146414051306656745055079e-02 5.28179248387344695003786654297073e-03
            2.43990646039493053187641180556966e-01 9.16390924818523237849188234349640e-03 2.52703844879229944445087063797928e-03
            2.43990646039493053187641180556966e-01 7.46845444712321726576931268937187e-01 2.52703844879229944445087063797928e-03
            9.16390924818523237849188234349640e-03 2.43990646039493053187641180556966e-01 2.52703844879229944445087063797928e-03
            9.16390924818523237849188234349640e-03 7.46845444712321726576931268937187e-01 2.52703844879229944445087063797928e-03
            7.46845444712321726576931268937187e-01 2.43990646039493053187641180556966e-01 2.52703844879229944445087063797928e-03
            7.46845444712321726576931268937187e-01 9.16390924818523237849188234349640e-03 2.52703844879229944445087063797928e-03
            1.79343210529389861240812109599574e-02 1.79846498894837506649091452004541e-03 3.20214265585712799105921888553894e-04
            1.79343210529389861240812109599574e-02 9.80267213958112559879509717575274e-01 3.20214265585712799105921888553894e-04
            1.79846498894837506649091452004541e-03 1.79343210529389861240812109599574e-02 3.20214265585712799105921888553894e-04
            1.79846498894837506649091452004541e-03 9.80267213958112559879509717575274e-01 3.20214265585712799105921888553894e-04
            9.80267213958112559879509717575274e-01 1.79343210529389861240812109599574e-02 3.20214265585712799105921888553894e-04
            9.80267213958112559879509717575274e-01 1.79846498894837506649091452004541e-03 3.20214265585712799105921888553894e-04
            ];
        
    case  23
        
        % ALG. DEG.:   23
        % PTS CARD.:  103
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 2.1e-15
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 1.26265301615181083083605528827320e-02
            3.90072687570319903693771834696236e-02 3.90072687570319903693771834696236e-02 1.95787012951646830632990159415385e-03
            3.90072687570319903693771834696236e-02 9.21985462485936047016821248689666e-01 1.95787012951646830632990159415385e-03
            9.21985462485936047016821248689666e-01 3.90072687570319903693771834696236e-02 1.95787012951646830632990159415385e-03
            4.80328877337308501882517930425820e-01 4.80328877337308501882517930425820e-01 5.69894463390038189670727675206763e-03
            4.80328877337308501882517930425820e-01 3.93422453253829962349641391483601e-02 5.69894463390038189670727675206763e-03
            3.93422453253829962349641391483601e-02 4.80328877337308501882517930425820e-01 5.69894463390038189670727675206763e-03
            8.68410482076330536571617813024204e-02 8.68410482076330536571617813024204e-02 4.47995851275677370306738112049061e-03
            8.68410482076330536571617813024204e-02 8.26317903584733892685676437395159e-01 4.47995851275677370306738112049061e-03
            8.26317903584733892685676437395159e-01 8.68410482076330536571617813024204e-02 4.47995851275677370306738112049061e-03
            3.94323506011541535798414770397358e-01 3.94323506011541535798414770397358e-01 1.18373042315640143729016386942021e-02
            3.94323506011541535798414770397358e-01 2.11352987976916928403170459205285e-01 1.18373042315640143729016386942021e-02
            2.11352987976916928403170459205285e-01 3.94323506011541535798414770397358e-01 1.18373042315640143729016386942021e-02
            2.66251317877247206666879719705321e-01 2.66251317877247206666879719705321e-01 1.19039314437498853355146266608244e-02
            2.66251317877247206666879719705321e-01 4.67497364245505586666240560589358e-01 1.19039314437498853355146266608244e-02
            4.67497364245505586666240560589358e-01 2.66251317877247206666879719705321e-01 1.19039314437498853355146266608244e-02
            1.37129387311647477076803625095636e-01 1.37129387311647477076803625095636e-01 7.27972469637087546412423932906677e-03
            1.37129387311647477076803625095636e-01 7.25741225376705045846392749808729e-01 7.27972469637087546412423932906677e-03
            7.25741225376705045846392749808729e-01 1.37129387311647477076803625095636e-01 7.27972469637087546412423932906677e-03
            4.98959431209586301569913757703034e-01 4.98959431209586301569913757703034e-01 1.20377230209070540389859615970636e-03
            4.98959431209586301569913757703034e-01 2.08113758082739686017248459393159e-03 1.20377230209070540389859615970636e-03
            2.08113758082739686017248459393159e-03 4.98959431209586301569913757703034e-01 1.20377230209070540389859615970636e-03
            4.44692442127727516343327351933112e-01 4.44692442127727516343327351933112e-01 9.47597533466944599855885655870225e-03
            4.44692442127727516343327351933112e-01 1.10615115744544967313345296133775e-01 9.47597533466944599855885655870225e-03
            1.10615115744544967313345296133775e-01 4.44692442127727516343327351933112e-01 9.47597533466944599855885655870225e-03
            1.98749806396536110586481527207070e-01 1.98749806396536110586481527207070e-01 9.96763894005251588348315294751956e-03
            1.98749806396536110586481527207070e-01 6.02500387206927778827036945585860e-01 9.96763894005251588348315294751956e-03
            6.02500387206927778827036945585860e-01 1.98749806396536110586481527207070e-01 9.96763894005251588348315294751956e-03
            9.01644020559819528437550673061196e-03 9.01644020559819528437550673061196e-03 5.32680616414657709216384784411957e-04
            9.01644020559819528437550673061196e-03 9.81967119588803560858991659188177e-01 5.32680616414657709216384784411957e-04
            9.81967119588803560858991659188177e-01 9.01644020559819528437550673061196e-03 5.32680616414657709216384784411957e-04
            1.59503798924757217436720679870632e-01 2.38702536543536174284962925185027e-02 1.26408302769113220324903590352505e-03
            1.59503798924757217436720679870632e-01 8.16625947420889231054275114729535e-01 1.26408302769113220324903590352505e-03
            2.38702536543536174284962925185027e-02 1.59503798924757217436720679870632e-01 1.26408302769113220324903590352505e-03
            2.38702536543536174284962925185027e-02 8.16625947420889231054275114729535e-01 1.26408302769113220324903590352505e-03
            8.16625947420889231054275114729535e-01 1.59503798924757217436720679870632e-01 1.26408302769113220324903590352505e-03
            8.16625947420889231054275114729535e-01 2.38702536543536174284962925185027e-02 1.26408302769113220324903590352505e-03
            1.14101360322364536736827744789480e-01 5.18982176084453739051127740822267e-03 1.11250986486225778883396575480447e-03
            1.14101360322364536736827744789480e-01 8.80708817916791009139387824689038e-01 1.11250986486225778883396575480447e-03
            5.18982176084453739051127740822267e-03 1.14101360322364536736827744789480e-01 1.11250986486225778883396575480447e-03
            5.18982176084453739051127740822267e-03 8.80708817916791009139387824689038e-01 1.11250986486225778883396575480447e-03
            8.80708817916791009139387824689038e-01 1.14101360322364536736827744789480e-01 1.11250986486225778883396575480447e-03
            8.80708817916791009139387824689038e-01 5.18982176084453739051127740822267e-03 1.11250986486225778883396575480447e-03
            9.55398781717348988262727971232380e-02 3.27410291887064164306764268985717e-02 2.66401521559739328307014893937321e-03
            9.55398781717348988262727971232380e-02 8.71719092639558601476323929091450e-01 2.66401521559739328307014893937321e-03
            3.27410291887064164306764268985717e-02 9.55398781717348988262727971232380e-02 2.66401521559739328307014893937321e-03
            3.27410291887064164306764268985717e-02 8.71719092639558601476323929091450e-01 2.66401521559739328307014893937321e-03
            8.71719092639558601476323929091450e-01 9.55398781717348988262727971232380e-02 2.66401521559739328307014893937321e-03
            8.71719092639558601476323929091450e-01 3.27410291887064164306764268985717e-02 2.66401521559739328307014893937321e-03
            3.11162268051701940141384739035857e-01 2.44759985596638019594850632643102e-03 1.14055183812791782749229696491966e-03
            3.11162268051701940141384739035857e-01 6.86390132092331683999475444579730e-01 1.14055183812791782749229696491966e-03
            2.44759985596638019594850632643102e-03 3.11162268051701940141384739035857e-01 1.14055183812791782749229696491966e-03
            2.44759985596638019594850632643102e-03 6.86390132092331683999475444579730e-01 1.14055183812791782749229696491966e-03
            6.86390132092331683999475444579730e-01 3.11162268051701940141384739035857e-01 1.14055183812791782749229696491966e-03
            6.86390132092331683999475444579730e-01 2.44759985596638019594850632643102e-03 1.14055183812791782749229696491966e-03
            2.05617232058052074039622425516427e-01 8.72528958530853666897275644487308e-03 2.05737517220804660977861999526795e-03
            2.05617232058052074039622425516427e-01 7.85657478356639460415067333087791e-01 2.05737517220804660977861999526795e-03
            8.72528958530853666897275644487308e-03 2.05617232058052074039622425516427e-01 2.05737517220804660977861999526795e-03
            8.72528958530853666897275644487308e-03 7.85657478356639460415067333087791e-01 2.05737517220804660977861999526795e-03
            7.85657478356639460415067333087791e-01 2.05617232058052074039622425516427e-01 2.05737517220804660977861999526795e-03
            7.85657478356639460415067333087791e-01 8.72528958530853666897275644487308e-03 2.05737517220804660977861999526795e-03
            4.72616294497252981132717764012341e-02 7.16253991024448470603669392176016e-03 9.76295663945363494280738070330017e-04
            4.72616294497252981132717764012341e-02 9.45575830640030146057029014627915e-01 9.76295663945363494280738070330017e-04
            7.16253991024448470603669392176016e-03 4.72616294497252981132717764012341e-02 9.76295663945363494280738070330017e-04
            7.16253991024448470603669392176016e-03 9.45575830640030146057029014627915e-01 9.76295663945363494280738070330017e-04
            9.45575830640030146057029014627915e-01 4.72616294497252981132717764012341e-02 9.76295663945363494280738070330017e-04
            9.45575830640030146057029014627915e-01 7.16253991024448470603669392176016e-03 9.76295663945363494280738070330017e-04
            3.58509593569625084619190147350309e-01 6.85269541872130100168547528483032e-02 7.49055669659958681810829617120362e-03
            3.58509593569625084619190147350309e-01 5.72963452243161919241742907615844e-01 7.49055669659958681810829617120362e-03
            6.85269541872130100168547528483032e-02 3.58509593569625084619190147350309e-01 7.49055669659958681810829617120362e-03
            6.85269541872130100168547528483032e-02 5.72963452243161919241742907615844e-01 7.49055669659958681810829617120362e-03
            5.72963452243161919241742907615844e-01 3.58509593569625084619190147350309e-01 7.49055669659958681810829617120362e-03
            5.72963452243161919241742907615844e-01 6.85269541872130100168547528483032e-02 7.49055669659958681810829617120362e-03
            2.40482772035012670741593865386676e-01 1.01728329327284261474062532215612e-01 8.06062081850857960196865548141432e-03
            2.40482772035012670741593865386676e-01 6.57788898637703067784343602397712e-01 8.06062081850857960196865548141432e-03
            1.01728329327284261474062532215612e-01 2.40482772035012670741593865386676e-01 8.06062081850857960196865548141432e-03
            1.01728329327284261474062532215612e-01 6.57788898637703067784343602397712e-01 8.06062081850857960196865548141432e-03
            6.57788898637703067784343602397712e-01 2.40482772035012670741593865386676e-01 8.06062081850857960196865548141432e-03
            6.57788898637703067784343602397712e-01 1.01728329327284261474062532215612e-01 8.06062081850857960196865548141432e-03
            1.72932303129223974291761578570004e-01 5.83515752375154614361996152638312e-02 5.23512824656503436560051056858356e-03
            1.72932303129223974291761578570004e-01 7.68716121633260529577569286630023e-01 5.23512824656503436560051056858356e-03
            5.83515752375154614361996152638312e-02 1.72932303129223974291761578570004e-01 5.23512824656503436560051056858356e-03
            5.83515752375154614361996152638312e-02 7.68716121633260529577569286630023e-01 5.23512824656503436560051056858356e-03
            7.68716121633260529577569286630023e-01 1.72932303129223974291761578570004e-01 5.23512824656503436560051056858356e-03
            7.68716121633260529577569286630023e-01 5.83515752375154614361996152638312e-02 5.23512824656503436560051056858356e-03
            3.16304307653838101543897209921852e-01 1.54830155405516284305988961023104e-01 1.04221979294844089430904787718646e-02
            3.16304307653838101543897209921852e-01 5.28865536940645641905689444683958e-01 1.04221979294844089430904787718646e-02
            1.54830155405516284305988961023104e-01 3.16304307653838101543897209921852e-01 1.04221979294844089430904787718646e-02
            1.54830155405516284305988961023104e-01 5.28865536940645641905689444683958e-01 1.04221979294844089430904787718646e-02
            5.28865536940645641905689444683958e-01 3.16304307653838101543897209921852e-01 1.04221979294844089430904787718646e-02
            5.28865536940645641905689444683958e-01 1.54830155405516284305988961023104e-01 1.04221979294844089430904787718646e-02
            3.97758576803007635191988811129704e-01 1.47589697299451757389610051518503e-02 3.54888941726091327114045448354318e-03
            3.97758576803007635191988811129704e-01 5.87482453467047194273220611648867e-01 3.54888941726091327114045448354318e-03
            1.47589697299451757389610051518503e-02 3.97758576803007635191988811129704e-01 3.54888941726091327114045448354318e-03
            1.47589697299451757389610051518503e-02 5.87482453467047194273220611648867e-01 3.54888941726091327114045448354318e-03
            5.87482453467047194273220611648867e-01 3.97758576803007635191988811129704e-01 3.54888941726091327114045448354318e-03
            5.87482453467047194273220611648867e-01 1.47589697299451757389610051518503e-02 3.54888941726091327114045448354318e-03
            2.78794169814102266169442145837820e-01 3.29937081925328057341317844475270e-02 5.08778732835352045377197427455940e-03
            2.78794169814102266169442145837820e-01 6.88212121993364900340850454085739e-01 5.08778732835352045377197427455940e-03
            3.29937081925328057341317844475270e-02 2.78794169814102266169442145837820e-01 5.08778732835352045377197427455940e-03
            3.29937081925328057341317844475270e-02 6.88212121993364900340850454085739e-01 5.08778732835352045377197427455940e-03
            6.88212121993364900340850454085739e-01 2.78794169814102266169442145837820e-01 5.08778732835352045377197427455940e-03
            6.88212121993364900340850454085739e-01 3.29937081925328057341317844475270e-02 5.08778732835352045377197427455940e-03
            ];
        
    case  24
        
        % ALG. DEG.:   24
        % PTS CARD.:  112
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 9.9e-16
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 6.27284492280015797821546286172634e-03
            4.18890974910602731373643337065005e-01 4.18890974910602731373643337065005e-01 6.55526635094261841268936663595923e-03
            4.18890974910602731373643337065005e-01 1.62218050178794537252713325869991e-01 6.55526635094261841268936663595923e-03
            1.62218050178794537252713325869991e-01 4.18890974910602731373643337065005e-01 6.55526635094261841268936663595923e-03
            1.62360633716926272551717147507588e-01 1.62360633716926272551717147507588e-01 5.18950802820009569893411338625810e-03
            1.62360633716926272551717147507588e-01 6.75278732566147454896565704984823e-01 5.18950802820009569893411338625810e-03
            6.75278732566147454896565704984823e-01 1.62360633716926272551717147507588e-01 5.18950802820009569893411338625810e-03
            4.09856290011169493991971535251650e-02 4.09856290011169493991971535251650e-02 1.91684986546459125260344347196906e-03
            4.09856290011169493991971535251650e-02 9.18028741997766073446030077320756e-01 1.91684986546459125260344347196906e-03
            9.18028741997766073446030077320756e-01 4.09856290011169493991971535251650e-02 1.91684986546459125260344347196906e-03
            6.73127088788825697629469857474760e-03 6.73127088788825697629469857474760e-03 3.08627252748321524028485285384704e-04
            6.73127088788825697629469857474760e-03 9.86537458224223451352941083314363e-01 3.08627252748321524028485285384704e-04
            9.86537458224223451352941083314363e-01 6.73127088788825697629469857474760e-03 3.08627252748321524028485285384704e-04
            4.96255277675735129161438408118556e-01 4.96255277675735129161438408118556e-01 2.17162336108534838319261695005480e-03
            4.96255277675735129161438408118556e-01 7.48944464852974167712318376288749e-03 2.17162336108534838319261695005480e-03
            7.48944464852974167712318376288749e-03 4.96255277675735129161438408118556e-01 2.17162336108534838319261695005480e-03
            2.64231315438272496010085887974128e-01 2.64231315438272496010085887974128e-01 1.02600043357549200462308647274767e-02
            2.64231315438272496010085887974128e-01 4.71537369123455007979828224051744e-01 1.02600043357549200462308647274767e-02
            4.71537369123455007979828224051744e-01 2.64231315438272496010085887974128e-01 1.02600043357549200462308647274767e-02
            4.80612561792503223401240575185511e-01 4.80612561792503223401240575185511e-01 5.17624738542630039028757948926796e-03
            4.80612561792503223401240575185511e-01 3.87748764149935531975188496289775e-02 5.17624738542630039028757948926796e-03
            3.87748764149935531975188496289775e-02 4.80612561792503223401240575185511e-01 5.17624738542630039028757948926796e-03
            9.63284955992150526693507117670379e-02 9.63284955992150526693507117670379e-02 5.01369653369445299195827914218171e-03
            9.63284955992150526693507117670379e-02 8.07343008801569839150147345208097e-01 5.01369653369445299195827914218171e-03
            8.07343008801569839150147345208097e-01 9.63284955992150526693507117670379e-02 5.01369653369445299195827914218171e-03
            3.75352926702086253651913239082205e-01 3.75352926702086253651913239082205e-01 9.49729325867632728774836436969053e-03
            3.75352926702086253651913239082205e-01 2.49294146595827492696173521835590e-01 9.49729325867632728774836436969053e-03
            2.49294146595827492696173521835590e-01 3.75352926702086253651913239082205e-01 9.49729325867632728774836436969053e-03
            2.41479760073594007518948956203531e-01 1.70367282462443730972623257002851e-01 7.07252290324242131880350115125111e-03
            2.41479760073594007518948956203531e-01 5.88152957463962233752852171164704e-01 7.07252290324242131880350115125111e-03
            1.70367282462443730972623257002851e-01 2.41479760073594007518948956203531e-01 7.07252290324242131880350115125111e-03
            1.70367282462443730972623257002851e-01 5.88152957463962233752852171164704e-01 7.07252290324242131880350115125111e-03
            5.88152957463962233752852171164704e-01 2.41479760073594007518948956203531e-01 7.07252290324242131880350115125111e-03
            5.88152957463962233752852171164704e-01 1.70367282462443730972623257002851e-01 7.07252290324242131880350115125111e-03
            3.28975808924226342799812528028269e-01 1.69759795860736056871687083003053e-01 7.63722130066231148087796398726823e-03
            3.28975808924226342799812528028269e-01 5.01264395215037628084076004597591e-01 7.63722130066231148087796398726823e-03
            1.69759795860736056871687083003053e-01 3.28975808924226342799812528028269e-01 7.63722130066231148087796398726823e-03
            1.69759795860736056871687083003053e-01 5.01264395215037628084076004597591e-01 7.63722130066231148087796398726823e-03
            5.01264395215037628084076004597591e-01 3.28975808924226342799812528028269e-01 7.63722130066231148087796398726823e-03
            5.01264395215037628084076004597591e-01 1.69759795860736056871687083003053e-01 7.63722130066231148087796398726823e-03
            9.31674097798811384762984744156711e-02 3.83182258210193957270739417708683e-02 2.68313572708388367246690187073455e-03
            9.31674097798811384762984744156711e-02 8.68514364399099458857733679906232e-01 2.68313572708388367246690187073455e-03
            3.83182258210193957270739417708683e-02 9.31674097798811384762984744156711e-02 2.68313572708388367246690187073455e-03
            3.83182258210193957270739417708683e-02 8.68514364399099458857733679906232e-01 2.68313572708388367246690187073455e-03
            8.68514364399099458857733679906232e-01 9.31674097798811384762984744156711e-02 2.68313572708388367246690187073455e-03
            8.68514364399099458857733679906232e-01 3.83182258210193957270739417708683e-02 2.68313572708388367246690187073455e-03
            3.94520279800194328956308709166478e-01 9.26564815207575470656919947032293e-02 7.51592717487066953180319117677755e-03
            3.94520279800194328956308709166478e-01 5.12823238679048110100211488315836e-01 7.51592717487066953180319117677755e-03
            9.26564815207575470656919947032293e-02 3.94520279800194328956308709166478e-01 7.51592717487066953180319117677755e-03
            9.26564815207575470656919947032293e-02 5.12823238679048110100211488315836e-01 7.51592717487066953180319117677755e-03
            5.12823238679048110100211488315836e-01 3.94520279800194328956308709166478e-01 7.51592717487066953180319117677755e-03
            5.12823238679048110100211488315836e-01 9.26564815207575470656919947032293e-02 7.51592717487066953180319117677755e-03
            1.62677416394477414307573326368583e-01 4.11887142484753873694636183699913e-02 3.60206708739871329580384262669668e-03
            1.62677416394477414307573326368583e-01 7.96133869357047219139644766983110e-01 3.60206708739871329580384262669668e-03
            4.11887142484753873694636183699913e-02 1.62677416394477414307573326368583e-01 3.60206708739871329580384262669668e-03
            4.11887142484753873694636183699913e-02 7.96133869357047219139644766983110e-01 3.60206708739871329580384262669668e-03
            7.96133869357047219139644766983110e-01 1.62677416394477414307573326368583e-01 3.60206708739871329580384262669668e-03
            7.96133869357047219139644766983110e-01 4.11887142484753873694636183699913e-02 3.60206708739871329580384262669668e-03
            2.53589014218879471584955354046542e-01 3.95709049701580550806312430722755e-02 4.45243846408178208656769569984135e-03
            2.53589014218879471584955354046542e-01 7.06840080810962501089989018510096e-01 4.45243846408178208656769569984135e-03
            3.95709049701580550806312430722755e-02 2.53589014218879471584955354046542e-01 4.45243846408178208656769569984135e-03
            3.95709049701580550806312430722755e-02 7.06840080810962501089989018510096e-01 4.45243846408178208656769569984135e-03
            7.06840080810962501089989018510096e-01 2.53589014218879471584955354046542e-01 4.45243846408178208656769569984135e-03
            7.06840080810962501089989018510096e-01 3.95709049701580550806312430722755e-02 4.45243846408178208656769569984135e-03
            3.62252241317791268127024295608862e-01 3.85927001748961398530646249582787e-02 4.97362593784120895351597013700484e-03
            3.62252241317791268127024295608862e-01 5.99155058507312543447653752082260e-01 4.97362593784120895351597013700484e-03
            3.85927001748961398530646249582787e-02 3.62252241317791268127024295608862e-01 4.97362593784120895351597013700484e-03
            3.85927001748961398530646249582787e-02 5.99155058507312543447653752082260e-01 4.97362593784120895351597013700484e-03
            5.99155058507312543447653752082260e-01 3.62252241317791268127024295608862e-01 4.97362593784120895351597013700484e-03
            5.99155058507312543447653752082260e-01 3.85927001748961398530646249582787e-02 4.97362593784120895351597013700484e-03
            2.81622577706160837074378378019901e-01 9.45349617365990324557856183673721e-02 7.17617578907872588622751663933741e-03
            2.81622577706160837074378378019901e-01 6.23842460557240130469836003612727e-01 7.17617578907872588622751663933741e-03
            9.45349617365990324557856183673721e-02 2.81622577706160837074378378019901e-01 7.17617578907872588622751663933741e-03
            9.45349617365990324557856183673721e-02 6.23842460557240130469836003612727e-01 7.17617578907872588622751663933741e-03
            6.23842460557240130469836003612727e-01 2.81622577706160837074378378019901e-01 7.17617578907872588622751663933741e-03
            6.23842460557240130469836003612727e-01 9.45349617365990324557856183673721e-02 7.17617578907872588622751663933741e-03
            3.83272664992659217730874843255151e-01 7.38799463229424056148753052752909e-03 2.12107463340188403047581466864813e-03
            3.83272664992659217730874843255151e-01 6.09339340375046556452787172020180e-01 2.12107463340188403047581466864813e-03
            7.38799463229424056148753052752909e-03 3.83272664992659217730874843255151e-01 2.12107463340188403047581466864813e-03
            7.38799463229424056148753052752909e-03 6.09339340375046556452787172020180e-01 2.12107463340188403047581466864813e-03
            6.09339340375046556452787172020180e-01 3.83272664992659217730874843255151e-01 2.12107463340188403047581466864813e-03
            6.09339340375046556452787172020180e-01 7.38799463229424056148753052752909e-03 2.12107463340188403047581466864813e-03
            2.73750352516260508917156357711065e-01 7.54600316231281847967338194393960e-03 2.04063753855822501911965183296616e-03
            2.73750352516260508917156357711065e-01 7.18703644321426637908700740808854e-01 2.04063753855822501911965183296616e-03
            7.54600316231281847967338194393960e-03 2.73750352516260508917156357711065e-01 2.04063753855822501911965183296616e-03
            7.54600316231281847967338194393960e-03 7.18703644321426637908700740808854e-01 2.04063753855822501911965183296616e-03
            7.18703644321426637908700740808854e-01 2.73750352516260508917156357711065e-01 2.04063753855822501911965183296616e-03
            7.18703644321426637908700740808854e-01 7.54600316231281847967338194393960e-03 2.04063753855822501911965183296616e-03
            9.41213427973660310410508600398316e-02 7.23455845778213976066117751884121e-03 1.29460619119899219761626874003468e-03
            9.41213427973660310410508600398316e-02 8.98644098744851849147607936174609e-01 1.29460619119899219761626874003468e-03
            7.23455845778213976066117751884121e-03 9.41213427973660310410508600398316e-02 1.29460619119899219761626874003468e-03
            7.23455845778213976066117751884121e-03 8.98644098744851849147607936174609e-01 1.29460619119899219761626874003468e-03
            8.98644098744851849147607936174609e-01 9.41213427973660310410508600398316e-02 1.29460619119899219761626874003468e-03
            8.98644098744851849147607936174609e-01 7.23455845778213976066117751884121e-03 1.29460619119899219761626874003468e-03
            1.80396151886765720995242645585677e-01 9.55662695273652579297518627754471e-02 5.92178107127155298888832035686391e-03
            1.80396151886765720995242645585677e-01 7.24037578585869034952793299453333e-01 5.92178107127155298888832035686391e-03
            9.55662695273652579297518627754471e-02 1.80396151886765720995242645585677e-01 5.92178107127155298888832035686391e-03
            9.55662695273652579297518627754471e-02 7.24037578585869034952793299453333e-01 5.92178107127155298888832035686391e-03
            7.24037578585869034952793299453333e-01 1.80396151886765720995242645585677e-01 5.92178107127155298888832035686391e-03
            7.24037578585869034952793299453333e-01 9.55662695273652579297518627754471e-02 5.92178107127155298888832035686391e-03
            1.74737346282805683372885141579900e-01 7.98792188084796773994700203047614e-03 1.85361338212315391454909629942449e-03
            1.74737346282805683372885141579900e-01 8.17274731836346357560785236273659e-01 1.85361338212315391454909629942449e-03
            7.98792188084796773994700203047614e-03 1.74737346282805683372885141579900e-01 1.85361338212315391454909629942449e-03
            7.98792188084796773994700203047614e-03 8.17274731836346357560785236273659e-01 1.85361338212315391454909629942449e-03
            8.17274731836346357560785236273659e-01 1.74737346282805683372885141579900e-01 1.85361338212315391454909629942449e-03
            8.17274731836346357560785236273659e-01 7.98792188084796773994700203047614e-03 1.85361338212315391454909629942449e-03
            3.72914720512912134520178142338409e-02 8.07491087020877768010151953603781e-03 8.98473792723288055585284528348211e-04
            3.72914720512912134520178142338409e-02 9.54633617078500029684562377951806e-01 8.98473792723288055585284528348211e-04
            8.07491087020877768010151953603781e-03 3.72914720512912134520178142338409e-02 8.98473792723288055585284528348211e-04
            8.07491087020877768010151953603781e-03 9.54633617078500029684562377951806e-01 8.98473792723288055585284528348211e-04
            9.54633617078500029684562377951806e-01 3.72914720512912134520178142338409e-02 8.98473792723288055585284528348211e-04
            9.54633617078500029684562377951806e-01 8.07491087020877768010151953603781e-03 8.98473792723288055585284528348211e-04
            ];
        
    case  25
        
        % ALG. DEG.:   25
        % PTS CARD.:  120
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.8e-15
        
        xyw=[
            3.87642030404563353584990181843750e-01 3.87642030404563353584990181843750e-01 6.84492577413611889142197597379891e-03
            3.87642030404563353584990181843750e-01 2.24715939190873292830019636312500e-01 6.84492577413611889142197597379891e-03
            2.24715939190873292830019636312500e-01 3.87642030404563353584990181843750e-01 6.84492577413611889142197597379891e-03
            2.11004508061496565129999680721085e-01 2.11004508061496565129999680721085e-01 5.79363161800529324896302796332748e-03
            2.11004508061496565129999680721085e-01 5.77990983877006869740000638557831e-01 5.79363161800529324896302796332748e-03
            5.77990983877006869740000638557831e-01 2.11004508061496565129999680721085e-01 5.79363161800529324896302796332748e-03
            2.99492315804508413279449996480253e-01 2.99492315804508413279449996480253e-01 9.00882035085073286373447132291403e-03
            2.99492315804508413279449996480253e-01 4.01015368390983173441100007039495e-01 9.00882035085073286373447132291403e-03
            4.01015368390983173441100007039495e-01 2.99492315804508413279449996480253e-01 9.00882035085073286373447132291403e-03
            3.72229259924406336024205188550695e-02 3.72229259924406336024205188550695e-02 1.69864886095236720942036612314041e-03
            3.72229259924406336024205188550695e-02 9.25554148015118705039583346660947e-01 1.69864886095236720942036612314041e-03
            9.25554148015118705039583346660947e-01 3.72229259924406336024205188550695e-02 1.69864886095236720942036612314041e-03
            1.45109243574500224038814621962956e-01 1.45109243574500224038814621962956e-01 5.74576293128239642959975697067421e-03
            1.45109243574500224038814621962956e-01 7.09781512850999551922370756074088e-01 5.74576293128239642959975697067421e-03
            7.09781512850999551922370756074088e-01 1.45109243574500224038814621962956e-01 5.74576293128239642959975697067421e-03
            4.24759304540574700315147538276506e-01 4.24759304540574700315147538276506e-01 7.95565506872920533143478394322301e-03
            4.24759304540574700315147538276506e-01 1.50481390918850599369704923446989e-01 7.95565506872920533143478394322301e-03
            1.50481390918850599369704923446989e-01 4.24759304540574700315147538276506e-01 7.95565506872920533143478394322301e-03
            4.62208708748706120772453687095549e-01 4.62208708748706120772453687095549e-01 6.82713759376400372569371199915622e-03
            4.62208708748706120772453687095549e-01 7.55825825025877584550926258089021e-02 6.82713759376400372569371199915622e-03
            7.55825825025877584550926258089021e-02 4.62208708748706120772453687095549e-01 6.82713759376400372569371199915622e-03
            9.29497017007697989843251207275898e-02 9.29497017007697989843251207275898e-02 4.59141062991001540349866871792983e-03
            9.29497017007697989843251207275898e-02 8.14100596598460457542500989802647e-01 4.59141062991001540349866871792983e-03
            8.14100596598460457542500989802647e-01 9.29497017007697989843251207275898e-02 4.59141062991001540349866871792983e-03
            7.83534428260363076390593306541632e-03 7.83534428260363076390593306541632e-03 4.03255144162308163269942129502965e-04
            7.83534428260363076390593306541632e-03 9.84329311434792741941635085822782e-01 4.03255144162308163269942129502965e-04
            9.84329311434792741941635085822782e-01 7.83534428260363076390593306541632e-03 4.03255144162308163269942129502965e-04
            4.89039369660395462080515471825493e-01 4.89039369660395462080515471825493e-01 4.22204297326053603484075082974414e-03
            4.89039369660395462080515471825493e-01 2.19212606792090758389690563490149e-02 4.22204297326053603484075082974414e-03
            2.19212606792090758389690563490149e-02 4.89039369660395462080515471825493e-01 4.22204297326053603484075082974414e-03
            4.40416927479343300877445699370583e-01 1.81886663427438814884251616632582e-03 8.37408915967352085810149198152885e-04
            4.40416927479343300877445699370583e-01 5.57764205886382313792637432925403e-01 8.37408915967352085810149198152885e-04
            1.81886663427438814884251616632582e-03 4.40416927479343300877445699370583e-01 8.37408915967352085810149198152885e-04
            1.81886663427438814884251616632582e-03 5.57764205886382313792637432925403e-01 8.37408915967352085810149198152885e-04
            5.57764205886382313792637432925403e-01 4.40416927479343300877445699370583e-01 8.37408915967352085810149198152885e-04
            5.57764205886382313792637432925403e-01 1.81886663427438814884251616632582e-03 8.37408915967352085810149198152885e-04
            1.59007906197327880670755462233501e-01 3.69601415796714863337690815114911e-02 3.15573901237963536439701250913004e-03
            1.59007906197327880670755462233501e-01 8.04031952223000612178793744533323e-01 3.15573901237963536439701250913004e-03
            3.69601415796714863337690815114911e-02 1.59007906197327880670755462233501e-01 3.15573901237963536439701250913004e-03
            3.69601415796714863337690815114911e-02 8.04031952223000612178793744533323e-01 3.15573901237963536439701250913004e-03
            8.04031952223000612178793744533323e-01 1.59007906197327880670755462233501e-01 3.15573901237963536439701250913004e-03
            8.04031952223000612178793744533323e-01 3.69601415796714863337690815114911e-02 3.15573901237963536439701250913004e-03
            1.77353796757252879512378740400891e-01 7.88580680056353011675440711769625e-02 4.75751078372788245574032117701790e-03
            1.77353796757252879512378740400891e-01 7.43788135237111736053350341535406e-01 4.75751078372788245574032117701790e-03
            7.88580680056353011675440711769625e-02 1.77353796757252879512378740400891e-01 4.75751078372788245574032117701790e-03
            7.88580680056353011675440711769625e-02 7.43788135237111736053350341535406e-01 4.75751078372788245574032117701790e-03
            7.43788135237111736053350341535406e-01 1.77353796757252879512378740400891e-01 4.75751078372788245574032117701790e-03
            7.43788135237111736053350341535406e-01 7.88580680056353011675440711769625e-02 4.75751078372788245574032117701790e-03
            2.70066735820959358260751059788163e-01 6.88475294314979391341324799213908e-02 5.44219680621845728479835102575635e-03
            2.70066735820959358260751059788163e-01 6.61085734747542619338389613403706e-01 5.44219680621845728479835102575635e-03
            6.88475294314979391341324799213908e-02 2.70066735820959358260751059788163e-01 5.44219680621845728479835102575635e-03
            6.88475294314979391341324799213908e-02 6.61085734747542619338389613403706e-01 5.44219680621845728479835102575635e-03
            6.61085734747542619338389613403706e-01 2.70066735820959358260751059788163e-01 5.44219680621845728479835102575635e-03
            6.61085734747542619338389613403706e-01 6.88475294314979391341324799213908e-02 5.44219680621845728479835102575635e-03
            3.41391033021149870041455187674728e-01 1.15999807640960211907454890933877e-01 7.92017614394921291853801648130684e-03
            3.41391033021149870041455187674728e-01 5.42609159337890001317816768278135e-01 7.92017614394921291853801648130684e-03
            1.15999807640960211907454890933877e-01 3.41391033021149870041455187674728e-01 7.92017614394921291853801648130684e-03
            1.15999807640960211907454890933877e-01 5.42609159337890001317816768278135e-01 7.92017614394921291853801648130684e-03
            5.42609159337890001317816768278135e-01 3.41391033021149870041455187674728e-01 7.92017614394921291853801648130684e-03
            5.42609159337890001317816768278135e-01 1.15999807640960211907454890933877e-01 7.92017614394921291853801648130684e-03
            3.73937979719584367988005624283687e-01 4.83174342873769688266882837979210e-02 5.32008534775438995945151532396267e-03
            3.73937979719584367988005624283687e-01 5.77744585993038684001987803640077e-01 5.32008534775438995945151532396267e-03
            4.83174342873769688266882837979210e-02 3.73937979719584367988005624283687e-01 5.32008534775438995945151532396267e-03
            4.83174342873769688266882837979210e-02 5.77744585993038684001987803640077e-01 5.32008534775438995945151532396267e-03
            5.77744585993038684001987803640077e-01 3.73937979719584367988005624283687e-01 5.32008534775438995945151532396267e-03
            5.77744585993038684001987803640077e-01 4.83174342873769688266882837979210e-02 5.32008534775438995945151532396267e-03
            9.91330633416821899173143606276426e-02 7.12831450125742636936276852566152e-03 1.27263581267450650788553012660032e-03
            9.91330633416821899173143606276426e-02 8.93738622157060458306432337849401e-01 1.27263581267450650788553012660032e-03
            7.12831450125742636936276852566152e-03 9.91330633416821899173143606276426e-02 1.27263581267450650788553012660032e-03
            7.12831450125742636936276852566152e-03 8.93738622157060458306432337849401e-01 1.27263581267450650788553012660032e-03
            8.93738622157060458306432337849401e-01 9.91330633416821899173143606276426e-02 1.27263581267450650788553012660032e-03
            8.93738622157060458306432337849401e-01 7.12831450125742636936276852566152e-03 1.27263581267450650788553012660032e-03
            2.99506418629674531217688127071597e-01 2.03692910584251041239411961214500e-01 8.95691044613802431206828202903125e-03
            2.99506418629674531217688127071597e-01 4.96800670786074427542899911713903e-01 8.95691044613802431206828202903125e-03
            2.03692910584251041239411961214500e-01 2.99506418629674531217688127071597e-01 8.95691044613802431206828202903125e-03
            2.03692910584251041239411961214500e-01 4.96800670786074427542899911713903e-01 8.95691044613802431206828202903125e-03
            4.96800670786074427542899911713903e-01 2.99506418629674531217688127071597e-01 8.95691044613802431206828202903125e-03
            4.96800670786074427542899911713903e-01 2.03692910584251041239411961214500e-01 8.95691044613802431206828202903125e-03
            1.78629848603616248059466897757375e-01 7.23616174794815877197029507783554e-03 1.63186984102462037324321464382137e-03
            1.78629848603616248059466897757375e-01 8.14133989648435640873458396526985e-01 1.63186984102462037324321464382137e-03
            7.23616174794815877197029507783554e-03 1.78629848603616248059466897757375e-01 1.63186984102462037324321464382137e-03
            7.23616174794815877197029507783554e-03 8.14133989648435640873458396526985e-01 1.63186984102462037324321464382137e-03
            8.14133989648435640873458396526985e-01 1.78629848603616248059466897757375e-01 1.63186984102462037324321464382137e-03
            8.14133989648435640873458396526985e-01 7.23616174794815877197029507783554e-03 1.63186984102462037324321464382137e-03
            3.62068801895972014470714839262655e-01 1.29138832500325321378520371240484e-02 2.72731918398721236845849702490341e-03
            3.62068801895972014470714839262655e-01 6.25017314853995475942838311311789e-01 2.72731918398721236845849702490341e-03
            1.29138832500325321378520371240484e-02 3.62068801895972014470714839262655e-01 2.72731918398721236845849702490341e-03
            1.29138832500325321378520371240484e-02 6.25017314853995475942838311311789e-01 2.72731918398721236845849702490341e-03
            6.25017314853995475942838311311789e-01 3.62068801895972014470714839262655e-01 2.72731918398721236845849702490341e-03
            6.25017314853995475942838311311789e-01 1.29138832500325321378520371240484e-02 2.72731918398721236845849702490341e-03
            8.87929154893665473213815175768104e-02 3.76879497842590865941403421857103e-02 2.63628096071470826786131702590410e-03
            8.87929154893665473213815175768104e-02 8.73519134726374435473417179309763e-01 2.63628096071470826786131702590410e-03
            3.76879497842590865941403421857103e-02 8.87929154893665473213815175768104e-02 2.63628096071470826786131702590410e-03
            3.76879497842590865941403421857103e-02 8.73519134726374435473417179309763e-01 2.63628096071470826786131702590410e-03
            8.73519134726374435473417179309763e-01 8.87929154893665473213815175768104e-02 2.63628096071470826786131702590410e-03
            8.73519134726374435473417179309763e-01 3.76879497842590865941403421857103e-02 2.63628096071470826786131702590410e-03
            2.33622810141715209208967962695169e-01 1.37006694087071001497690758696990e-01 6.87004129601127225962509115220200e-03
            2.33622810141715209208967962695169e-01 6.29370495771213733782190047350014e-01 6.87004129601127225962509115220200e-03
            1.37006694087071001497690758696990e-01 2.33622810141715209208967962695169e-01 6.87004129601127225962509115220200e-03
            1.37006694087071001497690758696990e-01 6.29370495771213733782190047350014e-01 6.87004129601127225962509115220200e-03
            6.29370495771213733782190047350014e-01 2.33622810141715209208967962695169e-01 6.87004129601127225962509115220200e-03
            6.29370495771213733782190047350014e-01 1.37006694087071001497690758696990e-01 6.87004129601127225962509115220200e-03
            2.56595409709019783850436624561553e-01 2.45400602475243957700801189503181e-02 3.65717045396642126409147977028624e-03
            2.56595409709019783850436624561553e-01 7.18864530043455851604505824070657e-01 3.65717045396642126409147977028624e-03
            2.45400602475243957700801189503181e-02 2.56595409709019783850436624561553e-01 3.65717045396642126409147977028624e-03
            2.45400602475243957700801189503181e-02 7.18864530043455851604505824070657e-01 3.65717045396642126409147977028624e-03
            7.18864530043455851604505824070657e-01 2.56595409709019783850436624561553e-01 3.65717045396642126409147977028624e-03
            7.18864530043455851604505824070657e-01 2.45400602475243957700801189503181e-02 3.65717045396642126409147977028624e-03
            4.10688191117846435052030074075446e-02 7.18882826169304090663247919223977e-03 8.46491817063665637557723364636786e-04
            4.10688191117846435052030074075446e-02 9.51742352626522292169397587713320e-01 8.46491817063665637557723364636786e-04
            7.18882826169304090663247919223977e-03 4.10688191117846435052030074075446e-02 8.46491817063665637557723364636786e-04
            7.18882826169304090663247919223977e-03 9.51742352626522292169397587713320e-01 8.46491817063665637557723364636786e-04
            9.51742352626522292169397587713320e-01 4.10688191117846435052030074075446e-02 8.46491817063665637557723364636786e-04
            9.51742352626522292169397587713320e-01 7.18882826169304090663247919223977e-03 8.46491817063665637557723364636786e-04
            2.79416188649260710530342066704179e-01 8.91464317498128172405358338181713e-04 7.55851039229439667505028310756643e-04
            2.79416188649260710530342066704179e-01 7.19692347033241142106874121964211e-01 7.55851039229439667505028310756643e-04
            8.91464317498128172405358338181713e-04 2.79416188649260710530342066704179e-01 7.55851039229439667505028310756643e-04
            8.91464317498128172405358338181713e-04 7.19692347033241142106874121964211e-01 7.55851039229439667505028310756643e-04
            7.19692347033241142106874121964211e-01 2.79416188649260710530342066704179e-01 7.55851039229439667505028310756643e-04
            7.19692347033241142106874121964211e-01 8.91464317498128172405358338181713e-04 7.55851039229439667505028310756643e-04
            ];
        
    case  26
        
        % ALG. DEG.:   26
        % PTS CARD.:  130
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 2.0e-15
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 1.02433312946116142350128086491168e-02
            6.67371225764660652712834121302876e-02 6.67371225764660652712834121302876e-02 2.45691265148300762993782697662937e-03
            6.67371225764660652712834121302876e-02 8.66525754847067841701857560110511e-01 2.45691265148300762993782697662937e-03
            8.66525754847067841701857560110511e-01 6.67371225764660652712834121302876e-02 2.45691265148300762993782697662937e-03
            6.34011649207668302741236487918286e-03 6.34011649207668302741236487918286e-03 2.63476558340935778333713468413180e-04
            6.34011649207668302741236487918286e-03 9.87319767015846672109091741731390e-01 2.63476558340935778333713468413180e-04
            9.87319767015846672109091741731390e-01 6.34011649207668302741236487918286e-03 2.63476558340935778333713468413180e-04
            4.93753032896384780325860219818424e-01 4.93753032896384780325860219818424e-01 2.65107959093367154951303987786559e-03
            4.93753032896384780325860219818424e-01 1.24939342072304393482795603631530e-02 2.65107959093367154951303987786559e-03
            1.24939342072304393482795603631530e-02 4.93753032896384780325860219818424e-01 2.65107959093367154951303987786559e-03
            3.88787497107593971179539948934689e-01 3.88787497107593971179539948934689e-01 9.73403391859143339581716958264224e-03
            3.88787497107593971179539948934689e-01 2.22425005784812057640920102130622e-01 9.73403391859143339581716958264224e-03
            2.22425005784812057640920102130622e-01 3.88787497107593971179539948934689e-01 9.73403391859143339581716958264224e-03
            2.73147100929078678888117792666890e-01 2.73147100929078678888117792666890e-01 9.76782346162376538534211078967928e-03
            2.73147100929078678888117792666890e-01 4.53705798141842642223764414666221e-01 9.76782346162376538534211078967928e-03
            4.53705798141842642223764414666221e-01 2.73147100929078678888117792666890e-01 9.76782346162376538534211078967928e-03
            4.71828563321166016830687794936239e-01 4.71828563321166016830687794936239e-01 5.76425181732844339049792736773270e-03
            4.71828563321166016830687794936239e-01 5.63428733576679663386244101275224e-02 5.76425181732844339049792736773270e-03
            5.63428733576679663386244101275224e-02 4.71828563321166016830687794936239e-01 5.76425181732844339049792736773270e-03
            1.54201430364544173068352961308847e-01 1.54201430364544173068352961308847e-01 6.62762972427263091779181536367105e-03
            1.54201430364544173068352961308847e-01 6.91597139270911709374445308640134e-01 6.62762972427263091779181536367105e-03
            6.91597139270911709374445308640134e-01 1.54201430364544173068352961308847e-01 6.62762972427263091779181536367105e-03
            2.12043163302205517339871221338399e-01 2.12043163302205517339871221338399e-01 8.47217253926404144370732041124938e-03
            2.12043163302205517339871221338399e-01 5.75913673395588965320257557323202e-01 8.47217253926404144370732041124938e-03
            5.75913673395588965320257557323202e-01 2.12043163302205517339871221338399e-01 8.47217253926404144370732041124938e-03
            4.35985419384383177110464657744160e-01 4.35985419384383177110464657744160e-01 8.20620030129394668894082798260570e-03
            4.35985419384383177110464657744160e-01 1.28029161231233645779070684511680e-01 8.20620030129394668894082798260570e-03
            1.28029161231233645779070684511680e-01 4.35985419384383177110464657744160e-01 8.20620030129394668894082798260570e-03
            8.00716549403165389620085079513956e-02 4.79466097543667854607996048343921e-03 6.99263224080135621305021054183726e-04
            8.00716549403165389620085079513956e-02 9.15133684084246779022464579611551e-01 6.99263224080135621305021054183726e-04
            4.79466097543667854607996048343921e-03 8.00716549403165389620085079513956e-02 6.99263224080135621305021054183726e-04
            4.79466097543667854607996048343921e-03 9.15133684084246779022464579611551e-01 6.99263224080135621305021054183726e-04
            9.15133684084246779022464579611551e-01 8.00716549403165389620085079513956e-02 6.99263224080135621305021054183726e-04
            9.15133684084246779022464579611551e-01 4.79466097543667854607996048343921e-03 6.99263224080135621305021054183726e-04
            3.16436115715307691997892902691092e-02 2.91551962068358448687543926780563e-02 6.02782386858442364822863179796286e-04
            3.16436115715307691997892902691092e-02 9.39201192221633451850948404171504e-01 6.02782386858442364822863179796286e-04
            2.91551962068358448687543926780563e-02 3.16436115715307691997892902691092e-02 6.02782386858442364822863179796286e-04
            2.91551962068358448687543926780563e-02 9.39201192221633451850948404171504e-01 6.02782386858442364822863179796286e-04
            9.39201192221633451850948404171504e-01 3.16436115715307691997892902691092e-02 6.02782386858442364822863179796286e-04
            9.39201192221633451850948404171504e-01 2.91551962068358448687543926780563e-02 6.02782386858442364822863179796286e-04
            7.53800475153986598852995371089492e-02 2.62093640224986615394620059760200e-02 1.65277235648383404031935839384460e-03
            7.53800475153986598852995371089492e-02 8.98410588462102754903071399894543e-01 1.65277235648383404031935839384460e-03
            2.62093640224986615394620059760200e-02 7.53800475153986598852995371089492e-02 1.65277235648383404031935839384460e-03
            2.62093640224986615394620059760200e-02 8.98410588462102754903071399894543e-01 1.65277235648383404031935839384460e-03
            8.98410588462102754903071399894543e-01 7.53800475153986598852995371089492e-02 1.65277235648383404031935839384460e-03
            8.98410588462102754903071399894543e-01 2.62093640224986615394620059760200e-02 1.65277235648383404031935839384460e-03
            3.31000343360322718533339525492920e-02 5.69811791687521804888882925865801e-03 5.42853671498377018771186452994471e-04
            3.31000343360322718533339525492920e-02 9.61201847747092497087351148365997e-01 5.42853671498377018771186452994471e-04
            5.69811791687521804888882925865801e-03 3.31000343360322718533339525492920e-02 5.42853671498377018771186452994471e-04
            5.69811791687521804888882925865801e-03 9.61201847747092497087351148365997e-01 5.42853671498377018771186452994471e-04
            9.61201847747092497087351148365997e-01 3.31000343360322718533339525492920e-02 5.42853671498377018771186452994471e-04
            9.61201847747092497087351148365997e-01 5.69811791687521804888882925865801e-03 5.42853671498377018771186452994471e-04
            1.32486189614567290151470047021576e-01 4.17247227421209468389839969404420e-02 3.20179894985640712753993675221409e-03
            1.32486189614567290151470047021576e-01 8.25789087643311825459591091203038e-01 3.20179894985640712753993675221409e-03
            4.17247227421209468389839969404420e-02 1.32486189614567290151470047021576e-01 3.20179894985640712753993675221409e-03
            4.17247227421209468389839969404420e-02 8.25789087643311825459591091203038e-01 3.20179894985640712753993675221409e-03
            8.25789087643311825459591091203038e-01 1.32486189614567290151470047021576e-01 3.20179894985640712753993675221409e-03
            8.25789087643311825459591091203038e-01 4.17247227421209468389839969404420e-02 3.20179894985640712753993675221409e-03
            1.08687132914402101446782467064622e-01 1.00045659106527559667476623417315e-01 2.30710553818915756890683965707467e-03
            1.08687132914402101446782467064622e-01 7.91267207979070352763528717332520e-01 2.30710553818915756890683965707467e-03
            1.00045659106527559667476623417315e-01 1.08687132914402101446782467064622e-01 2.30710553818915756890683965707467e-03
            1.00045659106527559667476623417315e-01 7.91267207979070352763528717332520e-01 2.30710553818915756890683965707467e-03
            7.91267207979070352763528717332520e-01 1.08687132914402101446782467064622e-01 2.30710553818915756890683965707467e-03
            7.91267207979070352763528717332520e-01 1.00045659106527559667476623417315e-01 2.30710553818915756890683965707467e-03
            2.50272313290526460249907358956989e-01 1.20614402205249035548995095723512e-01 7.18973661379936589133077262658844e-03
            2.50272313290526460249907358956989e-01 6.29113284504224545834460968762869e-01 7.18973661379936589133077262658844e-03
            1.20614402205249035548995095723512e-01 2.50272313290526460249907358956989e-01 7.18973661379936589133077262658844e-03
            1.20614402205249035548995095723512e-01 6.29113284504224545834460968762869e-01 7.18973661379936589133077262658844e-03
            6.29113284504224545834460968762869e-01 2.50272313290526460249907358956989e-01 7.18973661379936589133077262658844e-03
            6.29113284504224545834460968762869e-01 1.20614402205249035548995095723512e-01 7.18973661379936589133077262658844e-03
            3.89022062042761740396912273354246e-01 2.95379425169078370372321984405062e-02 4.12988360854341703815029163138206e-03
            3.89022062042761740396912273354246e-01 5.81439995440330315013000017643208e-01 4.12988360854341703815029163138206e-03
            2.95379425169078370372321984405062e-02 3.89022062042761740396912273354246e-01 4.12988360854341703815029163138206e-03
            2.95379425169078370372321984405062e-02 5.81439995440330315013000017643208e-01 4.12988360854341703815029163138206e-03
            5.81439995440330315013000017643208e-01 3.89022062042761740396912273354246e-01 4.12988360854341703815029163138206e-03
            5.81439995440330315013000017643208e-01 2.95379425169078370372321984405062e-02 4.12988360854341703815029163138206e-03
            3.58509296427661550143994872996700e-01 8.73784651638445236665475590598362e-02 6.86397910804284645813755361132280e-03
            3.58509296427661550143994872996700e-01 5.54112238408493995578396607015748e-01 6.86397910804284645813755361132280e-03
            8.73784651638445236665475590598362e-02 3.58509296427661550143994872996700e-01 6.86397910804284645813755361132280e-03
            8.73784651638445236665475590598362e-02 5.54112238408493995578396607015748e-01 6.86397910804284645813755361132280e-03
            5.54112238408493995578396607015748e-01 3.58509296427661550143994872996700e-01 6.86397910804284645813755361132280e-03
            5.54112238408493995578396607015748e-01 8.73784651638445236665475590598362e-02 6.86397910804284645813755361132280e-03
            1.86869179476221558555693036396406e-01 7.63119015129594124857703718589619e-02 5.19882276408715864046694576927621e-03
            1.86869179476221558555693036396406e-01 7.36818919010818973447385360486805e-01 5.19882276408715864046694576927621e-03
            7.63119015129594124857703718589619e-02 1.86869179476221558555693036396406e-01 5.19882276408715864046694576927621e-03
            7.63119015129594124857703718589619e-02 7.36818919010818973447385360486805e-01 5.19882276408715864046694576927621e-03
            7.36818919010818973447385360486805e-01 1.86869179476221558555693036396406e-01 5.19882276408715864046694576927621e-03
            7.36818919010818973447385360486805e-01 7.63119015129594124857703718589619e-02 5.19882276408715864046694576927621e-03
            4.14705909590306287526573214563541e-01 2.05753096537086583445175236306568e-03 9.28573735499041365455685781427064e-04
            4.14705909590306287526573214563541e-01 5.83236559444322844036889819108183e-01 9.28573735499041365455685781427064e-04
            2.05753096537086583445175236306568e-03 4.14705909590306287526573214563541e-01 9.28573735499041365455685781427064e-04
            2.05753096537086583445175236306568e-03 5.83236559444322844036889819108183e-01 9.28573735499041365455685781427064e-04
            5.83236559444322844036889819108183e-01 4.14705909590306287526573214563541e-01 9.28573735499041365455685781427064e-04
            5.83236559444322844036889819108183e-01 2.05753096537086583445175236306568e-03 9.28573735499041365455685781427064e-04
            3.19415305383438707931276212548255e-01 1.70478728497248960271903683860728e-01 8.79958359034759864114594307693551e-03
            3.19415305383438707931276212548255e-01 5.10105966119312248530093256704276e-01 8.79958359034759864114594307693551e-03
            1.70478728497248960271903683860728e-01 3.19415305383438707931276212548255e-01 8.79958359034759864114594307693551e-03
            1.70478728497248960271903683860728e-01 5.10105966119312248530093256704276e-01 8.79958359034759864114594307693551e-03
            5.10105966119312248530093256704276e-01 3.19415305383438707931276212548255e-01 8.79958359034759864114594307693551e-03
            5.10105966119312248530093256704276e-01 1.70478728497248960271903683860728e-01 8.79958359034759864114594307693551e-03
            1.43737626199764018108950835994619e-01 7.99960809148430283810338181638144e-03 1.48338083132825175255997685042075e-03
            1.43737626199764018108950835994619e-01 8.48262765708751698134904017933877e-01 1.48338083132825175255997685042075e-03
            7.99960809148430283810338181638144e-03 1.43737626199764018108950835994619e-01 1.48338083132825175255997685042075e-03
            7.99960809148430283810338181638144e-03 8.48262765708751698134904017933877e-01 1.48338083132825175255997685042075e-03
            8.48262765708751698134904017933877e-01 1.43737626199764018108950835994619e-01 1.48338083132825175255997685042075e-03
            8.48262765708751698134904017933877e-01 7.99960809148430283810338181638144e-03 1.48338083132825175255997685042075e-03
            2.83788138859470406849538903770735e-01 5.11658736851377873855462041774445e-02 5.05356221604433969440961504915322e-03
            2.83788138859470406849538903770735e-01 6.65045987455391784948233180330135e-01 5.05356221604433969440961504915322e-03
            5.11658736851377873855462041774445e-02 2.83788138859470406849538903770735e-01 5.05356221604433969440961504915322e-03
            5.11658736851377873855462041774445e-02 6.65045987455391784948233180330135e-01 5.05356221604433969440961504915322e-03
            6.65045987455391784948233180330135e-01 2.83788138859470406849538903770735e-01 5.05356221604433969440961504915322e-03
            6.65045987455391784948233180330135e-01 5.11658736851377873855462041774445e-02 5.05356221604433969440961504915322e-03
            2.16546666473477117387247403712536e-01 2.27845992508956714239776175645602e-02 3.13466892304028240529012450110713e-03
            2.16546666473477117387247403712536e-01 7.60668734275627200780434122862061e-01 3.13466892304028240529012450110713e-03
            2.27845992508956714239776175645602e-02 2.16546666473477117387247403712536e-01 3.13466892304028240529012450110713e-03
            2.27845992508956714239776175645602e-02 7.60668734275627200780434122862061e-01 3.13466892304028240529012450110713e-03
            7.60668734275627200780434122862061e-01 2.16546666473477117387247403712536e-01 3.13466892304028240529012450110713e-03
            7.60668734275627200780434122862061e-01 2.27845992508956714239776175645602e-02 3.13466892304028240529012450110713e-03
            3.12898503074879985597078757564304e-01 9.47329791221356194708125286751965e-03 2.29577919369931739840273721142694e-03
            3.12898503074879985597078757564304e-01 6.77628199012906495823926888988353e-01 2.29577919369931739840273721142694e-03
            9.47329791221356194708125286751965e-03 3.12898503074879985597078757564304e-01 2.29577919369931739840273721142694e-03
            9.47329791221356194708125286751965e-03 6.77628199012906495823926888988353e-01 2.29577919369931739840273721142694e-03
            6.77628199012906495823926888988353e-01 3.12898503074879985597078757564304e-01 2.29577919369931739840273721142694e-03
            6.77628199012906495823926888988353e-01 9.47329791221356194708125286751965e-03 2.29577919369931739840273721142694e-03
            2.26434797407717519623915336524078e-01 4.64007732175652793521480887761754e-04 5.69774457934106321604916800538376e-04
            2.26434797407717519623915336524078e-01 7.73101194860106755157858060556464e-01 5.69774457934106321604916800538376e-04
            4.64007732175652793521480887761754e-04 2.26434797407717519623915336524078e-01 5.69774457934106321604916800538376e-04
            4.64007732175652793521480887761754e-04 7.73101194860106755157858060556464e-01 5.69774457934106321604916800538376e-04
            7.73101194860106755157858060556464e-01 2.26434797407717519623915336524078e-01 5.69774457934106321604916800538376e-04
            7.73101194860106755157858060556464e-01 4.64007732175652793521480887761754e-04 5.69774457934106321604916800538376e-04
            ];
        
    case  27
        
        % ALG. DEG.:   27
        % PTS CARD.:  141
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.1e-15
        
        xyw=[
            3.80714021181187134423851148312679e-01 3.80714021181187134423851148312679e-01 4.78004248372996490973330097062899e-03
            3.80714021181187134423851148312679e-01 2.38571957637625731152297703374643e-01 4.78004248372996490973330097062899e-03
            2.38571957637625731152297703374643e-01 3.80714021181187134423851148312679e-01 4.78004248372996490973330097062899e-03
            4.46667803703864552122837494607666e-01 4.46667803703864552122837494607666e-01 4.70507990472711704055264192447794e-03
            4.46667803703864552122837494607666e-01 1.06664392592270895754325010784669e-01 4.70507990472711704055264192447794e-03
            1.06664392592270895754325010784669e-01 4.46667803703864552122837494607666e-01 4.70507990472711704055264192447794e-03
            4.16141378805412132102503619535128e-01 4.16141378805412132102503619535128e-01 6.02511351207522197898081017797267e-03
            4.16141378805412132102503619535128e-01 1.67717242389175735794992760929745e-01 6.02511351207522197898081017797267e-03
            1.67717242389175735794992760929745e-01 4.16141378805412132102503619535128e-01 6.02511351207522197898081017797267e-03
            8.03046477884382658984918634814676e-02 8.03046477884382658984918634814676e-02 2.60631093640094086555381025505085e-03
            8.03046477884382658984918634814676e-02 8.39390704423123468203016273037065e-01 2.60631093640094086555381025505085e-03
            8.39390704423123468203016273037065e-01 8.03046477884382658984918634814676e-02 2.60631093640094086555381025505085e-03
            2.33400406669870996712035093878512e-01 2.33400406669870996712035093878512e-01 6.73565769902469396041011506781615e-03
            2.33400406669870996712035093878512e-01 5.33199186660258006575929812242975e-01 6.73565769902469396041011506781615e-03
            5.33199186660258006575929812242975e-01 2.33400406669870996712035093878512e-01 6.73565769902469396041011506781615e-03
            3.01165465166509105365832965617301e-01 3.01165465166509105365832965617301e-01 7.87398289068133583334940794884460e-03
            3.01165465166509105365832965617301e-01 3.97669069666981789268334068765398e-01 7.87398289068133583334940794884460e-03
            3.97669069666981789268334068765398e-01 3.01165465166509105365832965617301e-01 7.87398289068133583334940794884460e-03
            1.74779966354899979341652738185076e-01 1.74779966354899979341652738185076e-01 5.64122127234919536198365364043639e-03
            1.74779966354899979341652738185076e-01 6.50440067290199985805543292372022e-01 5.64122127234919536198365364043639e-03
            6.50440067290199985805543292372022e-01 1.74779966354899979341652738185076e-01 5.64122127234919536198365364043639e-03
            4.85565054185162769950778738348163e-01 4.85565054185162769950778738348163e-01 3.55861870643732437421991043891012e-03
            4.85565054185162769950778738348163e-01 2.88698916296744600984425233036745e-02 3.55861870643732437421991043891012e-03
            2.88698916296744600984425233036745e-02 4.85565054185162769950778738348163e-01 3.55861870643732437421991043891012e-03
            3.25715201801814874249529907501710e-02 3.25715201801814874249529907501710e-02 1.38866976447709180100820169911913e-03
            3.25715201801814874249529907501710e-02 9.34856959639636997394518402870744e-01 1.38866976447709180100820169911913e-03
            9.34856959639636997394518402870744e-01 3.25715201801814874249529907501710e-02 1.38866976447709180100820169911913e-03
            1.27570901904677458382764143607346e-01 1.27570901904677458382764143607346e-01 4.87162246140887024437038377300269e-03
            1.27570901904677458382764143607346e-01 7.44858196190645083234471712785307e-01 4.87162246140887024437038377300269e-03
            7.44858196190645083234471712785307e-01 1.27570901904677458382764143607346e-01 4.87162246140887024437038377300269e-03
            6.63921918095859882325004974745752e-03 6.63921918095859882325004974745752e-03 2.87721202835251475739969384548544e-04
            6.63921918095859882325004974745752e-03 9.86721561638082778067371236829786e-01 2.87721202835251475739969384548544e-04
            9.86721561638082778067371236829786e-01 6.63921918095859882325004974745752e-03 2.87721202835251475739969384548544e-04
            2.87042196593496612422313774004579e-01 3.07306047272728651009732203647218e-02 2.76589741688336877151188630818979e-03
            2.87042196593496612422313774004579e-01 6.82227198679230539823947765398771e-01 2.76589741688336877151188630818979e-03
            3.07306047272728651009732203647218e-02 2.87042196593496612422313774004579e-01 2.76589741688336877151188630818979e-03
            3.07306047272728651009732203647218e-02 6.82227198679230539823947765398771e-01 2.76589741688336877151188630818979e-03
            6.82227198679230539823947765398771e-01 2.87042196593496612422313774004579e-01 2.76589741688336877151188630818979e-03
            6.82227198679230539823947765398771e-01 3.07306047272728651009732203647218e-02 2.76589741688336877151188630818979e-03
            3.45087841715568388600843263702700e-01 1.29152640063449730689626449020579e-01 6.27871810201827487696313312426355e-03
            3.45087841715568388600843263702700e-01 5.25759518220981880709530287276721e-01 6.27871810201827487696313312426355e-03
            1.29152640063449730689626449020579e-01 3.45087841715568388600843263702700e-01 6.27871810201827487696313312426355e-03
            1.29152640063449730689626449020579e-01 5.25759518220981880709530287276721e-01 6.27871810201827487696313312426355e-03
            5.25759518220981880709530287276721e-01 3.45087841715568388600843263702700e-01 6.27871810201827487696313312426355e-03
            5.25759518220981880709530287276721e-01 1.29152640063449730689626449020579e-01 6.27871810201827487696313312426355e-03
            3.75930157048661783836251970569720e-01 2.80334860952500125597097735408170e-02 3.19757634972720437838145102205090e-03
            3.75930157048661783836251970569720e-01 5.96036356856088245237401679332834e-01 3.19757634972720437838145102205090e-03
            2.80334860952500125597097735408170e-02 3.75930157048661783836251970569720e-01 3.19757634972720437838145102205090e-03
            2.80334860952500125597097735408170e-02 5.96036356856088245237401679332834e-01 3.19757634972720437838145102205090e-03
            5.96036356856088245237401679332834e-01 3.75930157048661783836251970569720e-01 3.19757634972720437838145102205090e-03
            5.96036356856088245237401679332834e-01 2.80334860952500125597097735408170e-02 3.19757634972720437838145102205090e-03
            3.16945588933131960907019220030634e-01 2.09130921137668762677108702519035e-01 6.85769661527542602474927235789437e-03
            3.16945588933131960907019220030634e-01 4.73923489929199304171447693079244e-01 6.85769661527542602474927235789437e-03
            2.09130921137668762677108702519035e-01 3.16945588933131960907019220030634e-01 6.85769661527542602474927235789437e-03
            2.09130921137668762677108702519035e-01 4.73923489929199304171447693079244e-01 6.85769661527542602474927235789437e-03
            4.73923489929199304171447693079244e-01 3.16945588933131960907019220030634e-01 6.85769661527542602474927235789437e-03
            4.73923489929199304171447693079244e-01 2.09130921137668762677108702519035e-01 6.85769661527542602474927235789437e-03
            4.07228393042719849503896512032952e-01 6.60389128497386734828467069746694e-02 4.93113505949478898859883457816977e-03
            4.07228393042719849503896512032952e-01 5.26732694107541421502105549734551e-01 4.93113505949478898859883457816977e-03
            6.60389128497386734828467069746694e-02 4.07228393042719849503896512032952e-01 4.93113505949478898859883457816977e-03
            6.60389128497386734828467069746694e-02 5.26732694107541421502105549734551e-01 4.93113505949478898859883457816977e-03
            5.26732694107541421502105549734551e-01 4.07228393042719849503896512032952e-01 4.93113505949478898859883457816977e-03
            5.26732694107541421502105549734551e-01 6.60389128497386734828467069746694e-02 4.93113505949478898859883457816977e-03
            2.13553598457823928802312707375677e-01 4.10305768191818395762560101047711e-02 3.22768645246484815053422678943207e-03
            2.13553598457823928802312707375677e-01 7.45415824722994169171386147354497e-01 3.22768645246484815053422678943207e-03
            4.10305768191818395762560101047711e-02 2.13553598457823928802312707375677e-01 3.22768645246484815053422678943207e-03
            4.10305768191818395762560101047711e-02 7.45415824722994169171386147354497e-01 3.22768645246484815053422678943207e-03
            7.45415824722994169171386147354497e-01 2.13553598457823928802312707375677e-01 3.22768645246484815053422678943207e-03
            7.45415824722994169171386147354497e-01 4.10305768191818395762560101047711e-02 3.22768645246484815053422678943207e-03
            3.28852878068892628959929425036535e-01 5.29964037179903642987843781497759e-03 1.46391318089955256229184321625780e-03
            3.28852878068892628959929425036535e-01 6.65847481559308285170573071809486e-01 1.46391318089955256229184321625780e-03
            5.29964037179903642987843781497759e-03 3.28852878068892628959929425036535e-01 1.46391318089955256229184321625780e-03
            5.29964037179903642987843781497759e-03 6.65847481559308285170573071809486e-01 1.46391318089955256229184321625780e-03
            6.65847481559308285170573071809486e-01 3.28852878068892628959929425036535e-01 1.46391318089955256229184321625780e-03
            6.65847481559308285170573071809486e-01 5.29964037179903642987843781497759e-03 1.46391318089955256229184321625780e-03
            1.39295306142148717087891895971552e-01 6.30739954149509007841345464839833e-02 3.56531765524351540105896241072969e-03
            1.39295306142148717087891895971552e-01 7.97630698442900465394700404431205e-01 3.56531765524351540105896241072969e-03
            6.30739954149509007841345464839833e-02 1.39295306142148717087891895971552e-01 3.56531765524351540105896241072969e-03
            6.30739954149509007841345464839833e-02 7.97630698442900465394700404431205e-01 3.56531765524351540105896241072969e-03
            7.97630698442900465394700404431205e-01 1.39295306142148717087891895971552e-01 3.56531765524351540105896241072969e-03
            7.97630698442900465394700404431205e-01 6.30739954149509007841345464839833e-02 3.56531765524351540105896241072969e-03
            2.55246254696978036591303862223867e-01 1.48962850938240159992531630450685e-01 6.17383156543068693095088761424449e-03
            2.55246254696978036591303862223867e-01 5.95790894364781831171740122954361e-01 6.17383156543068693095088761424449e-03
            1.48962850938240159992531630450685e-01 2.55246254696978036591303862223867e-01 6.17383156543068693095088761424449e-03
            1.48962850938240159992531630450685e-01 5.95790894364781831171740122954361e-01 6.17383156543068693095088761424449e-03
            5.95790894364781831171740122954361e-01 2.55246254696978036591303862223867e-01 6.17383156543068693095088761424449e-03
            5.95790894364781831171740122954361e-01 1.48962850938240159992531630450685e-01 6.17383156543068693095088761424449e-03
            2.08376015600374053793331086126273e-01 9.46970824331307137189384093289846e-02 5.34685029480813645008652201795485e-03
            2.08376015600374053793331086126273e-01 6.96926901966495315754457351431483e-01 5.34685029480813645008652201795485e-03
            9.46970824331307137189384093289846e-02 2.08376015600374053793331086126273e-01 5.34685029480813645008652201795485e-03
            9.46970824331307137189384093289846e-02 6.96926901966495315754457351431483e-01 5.34685029480813645008652201795485e-03
            6.96926901966495315754457351431483e-01 2.08376015600374053793331086126273e-01 5.34685029480813645008652201795485e-03
            6.96926901966495315754457351431483e-01 9.46970824331307137189384093289846e-02 5.34685029480813645008652201795485e-03
            4.40010551946215466845302444198751e-01 5.58071701526011761801404631455625e-03 1.62123379881967200413783736223650e-03
            4.40010551946215466845302444198751e-01 5.54408731038524504874942522292258e-01 1.62123379881967200413783736223650e-03
            5.58071701526011761801404631455625e-03 4.40010551946215466845302444198751e-01 1.62123379881967200413783736223650e-03
            5.58071701526011761801404631455625e-03 5.54408731038524504874942522292258e-01 1.62123379881967200413783736223650e-03
            5.54408731038524504874942522292258e-01 4.40010551946215466845302444198751e-01 1.62123379881967200413783736223650e-03
            5.54408731038524504874942522292258e-01 5.58071701526011761801404631455625e-03 1.62123379881967200413783736223650e-03
            3.02220941227821116026319714364945e-01 7.50769024331962436802356819498527e-02 5.46530554645664745638811155004078e-03
            3.02220941227821116026319714364945e-01 6.22702156338982626415656795870746e-01 5.46530554645664745638811155004078e-03
            7.50769024331962436802356819498527e-02 3.02220941227821116026319714364945e-01 5.46530554645664745638811155004078e-03
            7.50769024331962436802356819498527e-02 6.22702156338982626415656795870746e-01 5.46530554645664745638811155004078e-03
            6.22702156338982626415656795870746e-01 3.02220941227821116026319714364945e-01 5.46530554645664745638811155004078e-03
            6.22702156338982626415656795870746e-01 7.50769024331962436802356819498527e-02 5.46530554645664745638811155004078e-03
            8.19468025835336788542662134204875e-02 6.98252932445901815239341914320903e-03 1.00756156364485206268499783988091e-03
            8.19468025835336788542662134204875e-02 9.11070668092007318605851651227567e-01 1.00756156364485206268499783988091e-03
            6.98252932445901815239341914320903e-03 8.19468025835336788542662134204875e-02 1.00756156364485206268499783988091e-03
            6.98252932445901815239341914320903e-03 9.11070668092007318605851651227567e-01 1.00756156364485206268499783988091e-03
            9.11070668092007318605851651227567e-01 8.19468025835336788542662134204875e-02 1.00756156364485206268499783988091e-03
            9.11070668092007318605851651227567e-01 6.98252932445901815239341914320903e-03 1.00756156364485206268499783988091e-03
            3.43649699121419890945894337619393e-02 6.09356940376483413590591453612433e-03 5.98386804236581966269514776257665e-04
            3.43649699121419890945894337619393e-02 9.59541460684093183708398555609165e-01 5.98386804236581966269514776257665e-04
            6.09356940376483413590591453612433e-03 3.43649699121419890945894337619393e-02 5.98386804236581966269514776257665e-04
            6.09356940376483413590591453612433e-03 9.59541460684093183708398555609165e-01 5.98386804236581966269514776257665e-04
            9.59541460684093183708398555609165e-01 3.43649699121419890945894337619393e-02 5.98386804236581966269514776257665e-04
            9.59541460684093183708398555609165e-01 6.09356940376483413590591453612433e-03 5.98386804236581966269514776257665e-04
            8.01120738471011173409408456791425e-02 3.50344225276973952887260566058103e-02 2.16357901768035858908190149918482e-03
            8.01120738471011173409408456791425e-02 8.84853503625201542881484328972874e-01 2.16357901768035858908190149918482e-03
            3.50344225276973952887260566058103e-02 8.01120738471011173409408456791425e-02 2.16357901768035858908190149918482e-03
            3.50344225276973952887260566058103e-02 8.84853503625201542881484328972874e-01 2.16357901768035858908190149918482e-03
            8.84853503625201542881484328972874e-01 8.01120738471011173409408456791425e-02 2.16357901768035858908190149918482e-03
            8.84853503625201542881484328972874e-01 3.50344225276973952887260566058103e-02 2.16357901768035858908190149918482e-03
            1.47213431898922469942547763821494e-01 1.93520013180389742479814429998441e-02 2.31119355589055712474344517204372e-03
            1.47213431898922469942547763821494e-01 8.33434566783038621728962880297331e-01 2.31119355589055712474344517204372e-03
            1.93520013180389742479814429998441e-02 1.47213431898922469942547763821494e-01 2.31119355589055712474344517204372e-03
            1.93520013180389742479814429998441e-02 8.33434566783038621728962880297331e-01 2.31119355589055712474344517204372e-03
            8.33434566783038621728962880297331e-01 1.47213431898922469942547763821494e-01 2.31119355589055712474344517204372e-03
            8.33434566783038621728962880297331e-01 1.93520013180389742479814429998441e-02 2.31119355589055712474344517204372e-03
            2.29719653257843214877453874578350e-01 7.33247254904045766416009755062078e-03 1.69712686940351529900350424640010e-03
            2.29719653257843214877453874578350e-01 7.62947874193116382102175521140452e-01 1.69712686940351529900350424640010e-03
            7.33247254904045766416009755062078e-03 2.29719653257843214877453874578350e-01 1.69712686940351529900350424640010e-03
            7.33247254904045766416009755062078e-03 7.62947874193116382102175521140452e-01 1.69712686940351529900350424640010e-03
            7.62947874193116382102175521140452e-01 2.29719653257843214877453874578350e-01 1.69712686940351529900350424640010e-03
            7.62947874193116382102175521140452e-01 7.33247254904045766416009755062078e-03 1.69712686940351529900350424640010e-03
            1.47655521119869792556755783152767e-01 4.90328443462974511261875942835786e-04 4.23303067881925702917872422759160e-04
            1.47655521119869792556755783152767e-01 8.51854150436667234558285599632654e-01 4.23303067881925702917872422759160e-04
            4.90328443462974511261875942835786e-04 1.47655521119869792556755783152767e-01 4.23303067881925702917872422759160e-04
            4.90328443462974511261875942835786e-04 8.51854150436667234558285599632654e-01 4.23303067881925702917872422759160e-04
            8.51854150436667234558285599632654e-01 1.47655521119869792556755783152767e-01 4.23303067881925702917872422759160e-04
            8.51854150436667234558285599632654e-01 4.90328443462974511261875942835786e-04 4.23303067881925702917872422759160e-04
            ];
        
    case  28
        
        % ALG. DEG.:   28
        % PTS CARD.:  150
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.4e-15
        
        xyw=[
            3.03982922516484066477460146415979e-01 3.03982922516484066477460146415979e-01 7.18123315032306776956039229276030e-03
            3.03982922516484066477460146415979e-01 3.92034154967031867045079707168043e-01 7.18123315032306776956039229276030e-03
            3.92034154967031867045079707168043e-01 3.03982922516484066477460146415979e-01 7.18123315032306776956039229276030e-03
            4.80412619665786157913123233242914e-03 4.80412619665786157913123233242914e-03 1.55567604340748190470261613782554e-04
            4.80412619665786157913123233242914e-03 9.90391747606684247351438443729421e-01 1.55567604340748190470261613782554e-04
            9.90391747606684247351438443729421e-01 4.80412619665786157913123233242914e-03 1.55567604340748190470261613782554e-04
            4.58279904240411928295628740670509e-01 4.58279904240411928295628740670509e-01 4.42585250544658137528131192084402e-03
            4.58279904240411928295628740670509e-01 8.34401915191761434087425186589826e-02 4.42585250544658137528131192084402e-03
            8.34401915191761434087425186589826e-02 4.58279904240411928295628740670509e-01 4.42585250544658137528131192084402e-03
            3.86267973570042055797557623009197e-01 3.86267973570042055797557623009197e-01 7.10529319522409507497995306835037e-03
            3.86267973570042055797557623009197e-01 2.27464052859915888404884753981605e-01 7.10529319522409507497995306835037e-03
            2.27464052859915888404884753981605e-01 3.86267973570042055797557623009197e-01 7.10529319522409507497995306835037e-03
            2.58264072150462098420575784984976e-01 2.58264072150462098420575784984976e-01 6.54637429454448492582852026089313e-03
            2.58264072150462098420575784984976e-01 4.83471855699075803158848430030048e-01 6.54637429454448492582852026089313e-03
            4.83471855699075803158848430030048e-01 2.58264072150462098420575784984976e-01 6.54637429454448492582852026089313e-03
            1.05895844178627504539313974873949e-01 1.05895844178627504539313974873949e-01 3.90497168554317141200837859571493e-03
            1.05895844178627504539313974873949e-01 7.88208311642744963165796434623189e-01 3.90497168554317141200837859571493e-03
            7.88208311642744963165796434623189e-01 1.05895844178627504539313974873949e-01 3.90497168554317141200837859571493e-03
            4.29552202118899328908696588769089e-01 4.29552202118899328908696588769089e-01 6.77379780166275594771585488729215e-03
            4.29552202118899328908696588769089e-01 1.40895595762201342182606822461821e-01 6.77379780166275594771585488729215e-03
            1.40895595762201342182606822461821e-01 4.29552202118899328908696588769089e-01 6.77379780166275594771585488729215e-03
            4.84841132562589427479338155535515e-01 4.84841132562589427479338155535515e-01 3.84105528929751249833657311683055e-03
            4.84841132562589427479338155535515e-01 3.03177348748211450413236889289692e-02 3.84105528929751249833657311683055e-03
            3.03177348748211450413236889289692e-02 4.84841132562589427479338155535515e-01 3.84105528929751249833657311683055e-03
            1.58637688863059594934767915219709e-01 1.58637688863059594934767915219709e-01 5.97133482012411533762508142331171e-03
            1.58637688863059594934767915219709e-01 6.82724622273880754619312938302755e-01 5.97133482012411533762508142331171e-03
            6.82724622273880754619312938302755e-01 1.58637688863059594934767915219709e-01 5.97133482012411533762508142331171e-03
            6.08391923927586036890957643663569e-02 6.08391923927586036890957643663569e-02 2.56412600233903468713303475112752e-03
            6.08391923927586036890957643663569e-02 8.78321615214482820377384086896200e-01 2.56412600233903468713303475112752e-03
            8.78321615214482820377384086896200e-01 6.08391923927586036890957643663569e-02 2.56412600233903468713303475112752e-03
            4.55054005583463913597874750394112e-02 2.15243853694561254608075273608847e-02 1.05876977884044936803475955855447e-03
            4.55054005583463913597874750394112e-02 9.32970214072197490118298901506932e-01 1.05876977884044936803475955855447e-03
            2.15243853694561254608075273608847e-02 4.55054005583463913597874750394112e-02 1.05876977884044936803475955855447e-03
            2.15243853694561254608075273608847e-02 9.32970214072197490118298901506932e-01 1.05876977884044936803475955855447e-03
            9.32970214072197490118298901506932e-01 4.55054005583463913597874750394112e-02 1.05876977884044936803475955855447e-03
            9.32970214072197490118298901506932e-01 2.15243853694561254608075273608847e-02 1.05876977884044936803475955855447e-03
            2.13394454767087293767247047071578e-01 4.90696693575595060954341875003593e-02 2.94783611725712340126404065188126e-03
            2.13394454767087293767247047071578e-01 7.37535875875353186259530957613606e-01 2.94783611725712340126404065188126e-03
            4.90696693575595060954341875003593e-02 2.13394454767087293767247047071578e-01 2.94783611725712340126404065188126e-03
            4.90696693575595060954341875003593e-02 7.37535875875353186259530957613606e-01 2.94783611725712340126404065188126e-03
            7.37535875875353186259530957613606e-01 2.13394454767087293767247047071578e-01 2.94783611725712340126404065188126e-03
            7.37535875875353186259530957613606e-01 4.90696693575595060954341875003593e-02 2.94783611725712340126404065188126e-03
            2.42102511919319640654180147976149e-01 1.77658450296370312715410477721889e-01 6.27862810833844028901573608436593e-03
            2.42102511919319640654180147976149e-01 5.80239037784310074385984989930876e-01 6.27862810833844028901573608436593e-03
            1.77658450296370312715410477721889e-01 2.42102511919319640654180147976149e-01 6.27862810833844028901573608436593e-03
            1.77658450296370312715410477721889e-01 5.80239037784310074385984989930876e-01 6.27862810833844028901573608436593e-03
            5.80239037784310074385984989930876e-01 2.42102511919319640654180147976149e-01 6.27862810833844028901573608436593e-03
            5.80239037784310074385984989930876e-01 1.77658450296370312715410477721889e-01 6.27862810833844028901573608436593e-03
            3.27190732019169983058048956081620e-01 1.89812356292736877927396221821255e-01 6.55305706204969366096557692458191e-03
            3.27190732019169983058048956081620e-01 4.82996911688093111258979206468211e-01 6.55305706204969366096557692458191e-03
            1.89812356292736877927396221821255e-01 3.27190732019169983058048956081620e-01 6.55305706204969366096557692458191e-03
            1.89812356292736877927396221821255e-01 4.82996911688093111258979206468211e-01 6.55305706204969366096557692458191e-03
            4.82996911688093111258979206468211e-01 3.27190732019169983058048956081620e-01 6.55305706204969366096557692458191e-03
            4.82996911688093111258979206468211e-01 1.89812356292736877927396221821255e-01 6.55305706204969366096557692458191e-03
            1.41998166933174241322035413759295e-01 4.45838202328932217688439720859606e-03 8.89447709627756862536818172060293e-04
            1.41998166933174241322035413759295e-01 8.53543451043536460787208852707408e-01 8.89447709627756862536818172060293e-04
            4.45838202328932217688439720859606e-03 1.41998166933174241322035413759295e-01 8.89447709627756862536818172060293e-04
            4.45838202328932217688439720859606e-03 8.53543451043536460787208852707408e-01 8.89447709627756862536818172060293e-04
            8.53543451043536460787208852707408e-01 1.41998166933174241322035413759295e-01 8.89447709627756862536818172060293e-04
            8.53543451043536460787208852707408e-01 4.45838202328932217688439720859606e-03 8.89447709627756862536818172060293e-04
            1.75396393191461719140988861909136e-01 8.76779764843520492689776801853441e-02 4.00596464334748193769630830729511e-03
            1.75396393191461719140988861909136e-01 7.36925630324186231590033457905520e-01 4.00596464334748193769630830729511e-03
            8.76779764843520492689776801853441e-02 1.75396393191461719140988861909136e-01 4.00596464334748193769630830729511e-03
            8.76779764843520492689776801853441e-02 7.36925630324186231590033457905520e-01 4.00596464334748193769630830729511e-03
            7.36925630324186231590033457905520e-01 1.75396393191461719140988861909136e-01 4.00596464334748193769630830729511e-03
            7.36925630324186231590033457905520e-01 8.76779764843520492689776801853441e-02 4.00596464334748193769630830729511e-03
            3.92139613334414494083546287583886e-01 6.31803276344106712336667897034204e-02 4.23234786713830302140593175863614e-03
            3.92139613334414494083546287583886e-01 5.44680059031174890193938153970521e-01 4.23234786713830302140593175863614e-03
            6.31803276344106712336667897034204e-02 3.92139613334414494083546287583886e-01 4.23234786713830302140593175863614e-03
            6.31803276344106712336667897034204e-02 5.44680059031174890193938153970521e-01 4.23234786713830302140593175863614e-03
            5.44680059031174890193938153970521e-01 3.92139613334414494083546287583886e-01 4.23234786713830302140593175863614e-03
            5.44680059031174890193938153970521e-01 6.31803276344106712336667897034204e-02 4.23234786713830302140593175863614e-03
            3.34145615035921328406232078123139e-01 4.14946413392367364353363612394787e-03 1.18838213384389189582190926586236e-03
            3.34145615035921328406232078123139e-01 6.61704920830154974531467360066017e-01 1.18838213384389189582190926586236e-03
            4.14946413392367364353363612394787e-03 3.34145615035921328406232078123139e-01 1.18838213384389189582190926586236e-03
            4.14946413392367364353363612394787e-03 6.61704920830154974531467360066017e-01 1.18838213384389189582190926586236e-03
            6.61704920830154974531467360066017e-01 3.34145615035921328406232078123139e-01 1.18838213384389189582190926586236e-03
            6.61704920830154974531467360066017e-01 4.14946413392367364353363612394787e-03 1.18838213384389189582190926586236e-03
            1.74146196051182144959312836363097e-01 2.27948049259162482949037098478584e-02 2.13847116750897269996101179856396e-03
            1.74146196051182144959312836363097e-01 8.03058999022901676134722492861329e-01 2.13847116750897269996101179856396e-03
            2.27948049259162482949037098478584e-02 1.74146196051182144959312836363097e-01 2.13847116750897269996101179856396e-03
            2.27948049259162482949037098478584e-02 8.03058999022901676134722492861329e-01 2.13847116750897269996101179856396e-03
            8.03058999022901676134722492861329e-01 1.74146196051182144959312836363097e-01 2.13847116750897269996101179856396e-03
            8.03058999022901676134722492861329e-01 2.27948049259162482949037098478584e-02 2.13847116750897269996101179856396e-03
            2.75839008071824209888944778867881e-01 2.27008443717970145869777809366497e-02 2.56563869101864884536556843386279e-03
            2.75839008071824209888944778867881e-01 7.01460147556378754707395728473784e-01 2.56563869101864884536556843386279e-03
            2.27008443717970145869777809366497e-02 2.75839008071824209888944778867881e-01 2.56563869101864884536556843386279e-03
            2.27008443717970145869777809366497e-02 7.01460147556378754707395728473784e-01 2.56563869101864884536556843386279e-03
            7.01460147556378754707395728473784e-01 2.75839008071824209888944778867881e-01 2.56563869101864884536556843386279e-03
            7.01460147556378754707395728473784e-01 2.27008443717970145869777809366497e-02 2.56563869101864884536556843386279e-03
            2.62226671646522725700467759679668e-02 6.14964854266397079746164777702688e-03 4.74270212944700336223091685639019e-04
            2.62226671646522725700467759679668e-02 9.67627684292683754030406362289796e-01 4.74270212944700336223091685639019e-04
            6.14964854266397079746164777702688e-03 2.62226671646522725700467759679668e-02 4.74270212944700336223091685639019e-04
            6.14964854266397079746164777702688e-03 9.67627684292683754030406362289796e-01 4.74270212944700336223091685639019e-04
            9.67627684292683754030406362289796e-01 2.62226671646522725700467759679668e-02 4.74270212944700336223091685639019e-04
            9.67627684292683754030406362289796e-01 6.14964854266397079746164777702688e-03 4.74270212944700336223091685639019e-04
            2.43047202365926140466001470485935e-01 1.12033629342270985018714668512985e-01 5.12299328085210128880255808780930e-03
            2.43047202365926140466001470485935e-01 6.44919168291802846759708245372167e-01 5.12299328085210128880255808780930e-03
            1.12033629342270985018714668512985e-01 2.43047202365926140466001470485935e-01 5.12299328085210128880255808780930e-03
            1.12033629342270985018714668512985e-01 6.44919168291802846759708245372167e-01 5.12299328085210128880255808780930e-03
            6.44919168291802846759708245372167e-01 2.43047202365926140466001470485935e-01 5.12299328085210128880255808780930e-03
            6.44919168291802846759708245372167e-01 1.12033629342270985018714668512985e-01 5.12299328085210128880255808780930e-03
            2.29992984057907162709355475271877e-01 4.78148977298713338118796301046132e-03 1.16227268223202543154237709188692e-03
            2.29992984057907162709355475271877e-01 7.65225526169105774165757338778349e-01 1.16227268223202543154237709188692e-03
            4.78148977298713338118796301046132e-03 2.29992984057907162709355475271877e-01 1.16227268223202543154237709188692e-03
            4.78148977298713338118796301046132e-03 7.65225526169105774165757338778349e-01 1.16227268223202543154237709188692e-03
            7.65225526169105774165757338778349e-01 2.29992984057907162709355475271877e-01 1.16227268223202543154237709188692e-03
            7.65225526169105774165757338778349e-01 4.78148977298713338118796301046132e-03 1.16227268223202543154237709188692e-03
            2.95608280872401651961922652844805e-01 6.24487421796328864842529071665922e-02 4.46255807013827641288594705315518e-03
            2.95608280872401651961922652844805e-01 6.41942976947965426859354920452461e-01 4.46255807013827641288594705315518e-03
            6.24487421796328864842529071665922e-02 2.95608280872401651961922652844805e-01 4.46255807013827641288594705315518e-03
            6.24487421796328864842529071665922e-02 6.41942976947965426859354920452461e-01 4.46255807013827641288594705315518e-03
            6.41942976947965426859354920452461e-01 2.95608280872401651961922652844805e-01 4.46255807013827641288594705315518e-03
            6.41942976947965426859354920452461e-01 6.24487421796328864842529071665922e-02 4.46255807013827641288594705315518e-03
            1.21393450754091192234085383461206e-01 5.02111859134281163230539846153988e-02 2.95894188617694571530325475805512e-03
            1.21393450754091192234085383461206e-01 8.28395363332480649809497208480025e-01 2.95894188617694571530325475805512e-03
            5.02111859134281163230539846153988e-02 1.21393450754091192234085383461206e-01 2.95894188617694571530325475805512e-03
            5.02111859134281163230539846153988e-02 8.28395363332480649809497208480025e-01 2.95894188617694571530325475805512e-03
            8.28395363332480649809497208480025e-01 1.21393450754091192234085383461206e-01 2.95894188617694571530325475805512e-03
            8.28395363332480649809497208480025e-01 5.02111859134281163230539846153988e-02 2.95894188617694571530325475805512e-03
            3.73823800310209619635060107611935e-01 2.57279987428787430825405380119264e-02 3.13026732148427664498280265092944e-03
            3.73823800310209619635060107611935e-01 6.00448200946911536668437747721327e-01 3.13026732148427664498280265092944e-03
            2.57279987428787430825405380119264e-02 3.73823800310209619635060107611935e-01 3.13026732148427664498280265092944e-03
            2.57279987428787430825405380119264e-02 6.00448200946911536668437747721327e-01 3.13026732148427664498280265092944e-03
            6.00448200946911536668437747721327e-01 3.73823800310209619635060107611935e-01 3.13026732148427664498280265092944e-03
            6.00448200946911536668437747721327e-01 2.57279987428787430825405380119264e-02 3.13026732148427664498280265092944e-03
            4.42783406520243560944294358705520e-01 5.64656599346616185769454077103546e-03 1.57008882174401818293540955551180e-03
            4.42783406520243560944294358705520e-01 5.51570027486290248575073746906128e-01 1.57008882174401818293540955551180e-03
            5.64656599346616185769454077103546e-03 4.42783406520243560944294358705520e-01 1.57008882174401818293540955551180e-03
            5.64656599346616185769454077103546e-03 5.51570027486290248575073746906128e-01 1.57008882174401818293540955551180e-03
            5.51570027486290248575073746906128e-01 4.42783406520243560944294358705520e-01 1.57008882174401818293540955551180e-03
            5.51570027486290248575073746906128e-01 5.64656599346616185769454077103546e-03 1.57008882174401818293540955551180e-03
            3.37798663200582327537091487101861e-01 1.18089069715095071333443854655343e-01 6.39356921147790288606493902534567e-03
            3.37798663200582327537091487101861e-01 5.44112267084322698273979312943993e-01 6.39356921147790288606493902534567e-03
            1.18089069715095071333443854655343e-01 3.37798663200582327537091487101861e-01 6.39356921147790288606493902534567e-03
            1.18089069715095071333443854655343e-01 5.44112267084322698273979312943993e-01 6.39356921147790288606493902534567e-03
            5.44112267084322698273979312943993e-01 3.37798663200582327537091487101861e-01 6.39356921147790288606493902534567e-03
            5.44112267084322698273979312943993e-01 1.18089069715095071333443854655343e-01 6.39356921147790288606493902534567e-03
            9.22291891952822101563569390236808e-02 1.82422910122947215527666742218571e-02 1.60245809539651790608105752511392e-03
            9.22291891952822101563569390236808e-02 8.89528519792423089107558098476147e-01 1.60245809539651790608105752511392e-03
            1.82422910122947215527666742218571e-02 9.22291891952822101563569390236808e-02 1.60245809539651790608105752511392e-03
            1.82422910122947215527666742218571e-02 8.89528519792423089107558098476147e-01 1.60245809539651790608105752511392e-03
            8.89528519792423089107558098476147e-01 9.22291891952822101563569390236808e-02 1.60245809539651790608105752511392e-03
            8.89528519792423089107558098476147e-01 1.82422910122947215527666742218571e-02 1.60245809539651790608105752511392e-03
            7.02907404781327321829920151685656e-02 1.20025560148715230122939967571938e-03 3.62567297493041594286938833491263e-04
            7.02907404781327321829920151685656e-02 9.28509003920380049379446063539945e-01 3.62567297493041594286938833491263e-04
            1.20025560148715230122939967571938e-03 7.02907404781327321829920151685656e-02 3.62567297493041594286938833491263e-04
            1.20025560148715230122939967571938e-03 9.28509003920380049379446063539945e-01 3.62567297493041594286938833491263e-04
            9.28509003920380049379446063539945e-01 7.02907404781327321829920151685656e-02 3.62567297493041594286938833491263e-04
            9.28509003920380049379446063539945e-01 1.20025560148715230122939967571938e-03 3.62567297493041594286938833491263e-04
            ];
        
    case  29
        
        % ALG. DEG.:   29
        % PTS CARD.:  159
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.3e-15
        
        xyw=[
            4.98914824637686160624383546746685e-01 4.98914824637686160624383546746685e-01 7.58231051578450981204282399517069e-04
            4.98914824637686160624383546746685e-01 2.17035072462767875123290650662966e-03 7.58231051578450981204282399517069e-04
            2.17035072462767875123290650662966e-03 4.98914824637686160624383546746685e-01 7.58231051578450981204282399517069e-04
            4.34380426761730609186429319379386e-01 4.34380426761730609186429319379386e-01 5.58555014758392847501466960125072e-03
            4.34380426761730609186429319379386e-01 1.31239146476538781627141361241229e-01 5.58555014758392847501466960125072e-03
            1.31239146476538781627141361241229e-01 4.34380426761730609186429319379386e-01 5.58555014758392847501466960125072e-03
            4.10973356271179326992815106223134e-02 4.10973356271179326992815106223134e-02 1.43024575307844348248664800848928e-03
            4.10973356271179326992815106223134e-02 9.17805328745764148479224786569830e-01 1.43024575307844348248664800848928e-03
            9.17805328745764148479224786569830e-01 4.10973356271179326992815106223134e-02 1.43024575307844348248664800848928e-03
            2.08405305132400797685221505162190e-01 2.08405305132400797685221505162190e-01 6.26960172131161338343607880574382e-03
            2.08405305132400797685221505162190e-01 5.83189389735198404629556989675621e-01 6.26960172131161338343607880574382e-03
            5.83189389735198404629556989675621e-01 2.08405305132400797685221505162190e-01 6.26960172131161338343607880574382e-03
            1.60745884431963526051134749650373e-01 1.60745884431963526051134749650373e-01 5.28570892911378771572561419134217e-03
            1.60745884431963526051134749650373e-01 6.78508231136072947897730500699254e-01 5.28570892911378771572561419134217e-03
            6.78508231136072947897730500699254e-01 1.60745884431963526051134749650373e-01 5.28570892911378771572561419134217e-03
            4.88401602932602707518583429191494e-01 4.88401602932602707518583429191494e-01 3.06720058235945489122364904233109e-03
            4.88401602932602707518583429191494e-01 2.31967941347945849628331416170113e-02 3.06720058235945489122364904233109e-03
            2.31967941347945849628331416170113e-02 4.88401602932602707518583429191494e-01 3.06720058235945489122364904233109e-03
            3.02386411215128436680288359639235e-01 3.02386411215128436680288359639235e-01 8.15511940387160169652069896528701e-03
            3.02386411215128436680288359639235e-01 3.95227177569743126639423280721530e-01 8.15511940387160169652069896528701e-03
            3.95227177569743126639423280721530e-01 3.02386411215128436680288359639235e-01 8.15511940387160169652069896528701e-03
            1.14426812994425419756083783795475e-01 1.14426812994425419756083783795475e-01 4.08643922061356634767292916876613e-03
            1.14426812994425419756083783795475e-01 7.71146374011149160487832432409050e-01 4.08643922061356634767292916876613e-03
            7.71146374011149160487832432409050e-01 1.14426812994425419756083783795475e-01 4.08643922061356634767292916876613e-03
            4.64762431080738891431280990218511e-01 4.64762431080738891431280990218511e-01 5.15655831762924961314542571244601e-03
            4.64762431080738891431280990218511e-01 7.04751378385222171374380195629783e-02 5.15655831762924961314542571244601e-03
            7.04751378385222171374380195629783e-02 4.64762431080738891431280990218511e-01 5.15655831762924961314542571244601e-03
            7.37218813900996500221296514610003e-02 7.37218813900996500221296514610003e-02 2.80442356641565212690969666198271e-03
            7.37218813900996500221296514610003e-02 8.52556237219800672200165081449086e-01 2.80442356641565212690969666198271e-03
            8.52556237219800672200165081449086e-01 7.37218813900996500221296514610003e-02 2.80442356641565212690969666198271e-03
            3.90619178783263687471105640725000e-01 3.90619178783263687471105640725000e-01 7.95660764204445154768041703619019e-03
            3.90619178783263687471105640725000e-01 2.18761642433472625057788718550000e-01 7.95660764204445154768041703619019e-03
            2.18761642433472625057788718550000e-01 3.90619178783263687471105640725000e-01 7.95660764204445154768041703619019e-03
            5.89421088402292056573905654204282e-02 2.72874324792107044584943409404332e-03 3.84626485738124364744533467685983e-04
            5.89421088402292056573905654204282e-02 9.38329147911849736907186070311582e-01 3.84626485738124364744533467685983e-04
            2.72874324792107044584943409404332e-03 5.89421088402292056573905654204282e-02 3.84626485738124364744533467685983e-04
            2.72874324792107044584943409404332e-03 9.38329147911849736907186070311582e-01 3.84626485738124364744533467685983e-04
            9.38329147911849736907186070311582e-01 5.89421088402292056573905654204282e-02 3.84626485738124364744533467685983e-04
            9.38329147911849736907186070311582e-01 2.72874324792107044584943409404332e-03 3.84626485738124364744533467685983e-04
            3.49788010009331795657772090635262e-01 1.57177699867193487737182522323565e-01 5.22610734846121714747280151414088e-03
            3.49788010009331795657772090635262e-01 4.93034290123474716605045387041173e-01 5.22610734846121714747280151414088e-03
            1.57177699867193487737182522323565e-01 3.49788010009331795657772090635262e-01 5.22610734846121714747280151414088e-03
            1.57177699867193487737182522323565e-01 4.93034290123474716605045387041173e-01 5.22610734846121714747280151414088e-03
            4.93034290123474716605045387041173e-01 3.49788010009331795657772090635262e-01 5.22610734846121714747280151414088e-03
            4.93034290123474716605045387041173e-01 1.57177699867193487737182522323565e-01 5.22610734846121714747280151414088e-03
            3.23001823543550170558802392406506e-01 2.10096664482755952213510752812908e-03 6.76411974626204203983603768790545e-04
            3.23001823543550170558802392406506e-01 6.74897209811622378339279748615809e-01 6.76411974626204203983603768790545e-04
            2.10096664482755952213510752812908e-03 3.23001823543550170558802392406506e-01 6.76411974626204203983603768790545e-04
            2.10096664482755952213510752812908e-03 6.74897209811622378339279748615809e-01 6.76411974626204203983603768790545e-04
            6.74897209811622378339279748615809e-01 3.23001823543550170558802392406506e-01 6.76411974626204203983603768790545e-04
            6.74897209811622378339279748615809e-01 2.10096664482755952213510752812908e-03 6.76411974626204203983603768790545e-04
            1.58145854249516104594874832400819e-01 6.81658088137464396316644865692069e-02 3.21956835376690240005381760113323e-03
            1.58145854249516104594874832400819e-01 7.73688336936737441895672873215517e-01 3.21956835376690240005381760113323e-03
            6.81658088137464396316644865692069e-02 1.58145854249516104594874832400819e-01 3.21956835376690240005381760113323e-03
            6.81658088137464396316644865692069e-02 7.73688336936737441895672873215517e-01 3.21956835376690240005381760113323e-03
            7.73688336936737441895672873215517e-01 1.58145854249516104594874832400819e-01 3.21956835376690240005381760113323e-03
            7.73688336936737441895672873215517e-01 6.81658088137464396316644865692069e-02 3.21956835376690240005381760113323e-03
            2.95494682613533683757101044875526e-02 1.08309586036093533772950436855353e-02 6.36097218585815536544270099028608e-04
            2.95494682613533683757101044875526e-02 9.59619573135037207123332336777821e-01 6.36097218585815536544270099028608e-04
            1.08309586036093533772950436855353e-02 2.95494682613533683757101044875526e-02 6.36097218585815536544270099028608e-04
            1.08309586036093533772950436855353e-02 9.59619573135037207123332336777821e-01 6.36097218585815536544270099028608e-04
            9.59619573135037207123332336777821e-01 2.95494682613533683757101044875526e-02 6.36097218585815536544270099028608e-04
            9.59619573135037207123332336777821e-01 1.08309586036093533772950436855353e-02 6.36097218585815536544270099028608e-04
            2.91819173426537070703545850847149e-01 2.18932341980172556761274904602033e-01 6.77262328600378594317676927971661e-03
            2.91819173426537070703545850847149e-01 4.89248484593290400290754860179732e-01 6.77262328600378594317676927971661e-03
            2.18932341980172556761274904602033e-01 2.91819173426537070703545850847149e-01 6.77262328600378594317676927971661e-03
            2.18932341980172556761274904602033e-01 4.89248484593290400290754860179732e-01 6.77262328600378594317676927971661e-03
            4.89248484593290400290754860179732e-01 2.91819173426537070703545850847149e-01 6.77262328600378594317676927971661e-03
            4.89248484593290400290754860179732e-01 2.18932341980172556761274904602033e-01 6.77262328600378594317676927971661e-03
            7.55221785129957801085964774756576e-02 2.12868962407332593111952689923783e-02 1.38076537479522192843062189382408e-03
            7.55221785129957801085964774756576e-02 9.03190925246270981396889965253649e-01 1.38076537479522192843062189382408e-03
            2.12868962407332593111952689923783e-02 7.55221785129957801085964774756576e-02 1.38076537479522192843062189382408e-03
            2.12868962407332593111952689923783e-02 9.03190925246270981396889965253649e-01 1.38076537479522192843062189382408e-03
            9.03190925246270981396889965253649e-01 7.55221785129957801085964774756576e-02 1.38076537479522192843062189382408e-03
            9.03190925246270981396889965253649e-01 2.12868962407332593111952689923783e-02 1.38076537479522192843062189382408e-03
            1.17116669508898874063618222862715e-01 4.08472165761024560981518050084560e-02 2.25634342274366479080827119219066e-03
            1.17116669508898874063618222862715e-01 8.42036113914998662899336068221601e-01 2.25634342274366479080827119219066e-03
            4.08472165761024560981518050084560e-02 1.17116669508898874063618222862715e-01 2.25634342274366479080827119219066e-03
            4.08472165761024560981518050084560e-02 8.42036113914998662899336068221601e-01 2.25634342274366479080827119219066e-03
            8.42036113914998662899336068221601e-01 1.17116669508898874063618222862715e-01 2.25634342274366479080827119219066e-03
            8.42036113914998662899336068221601e-01 4.08472165761024560981518050084560e-02 2.25634342274366479080827119219066e-03
            1.11662181081691909423359021502620e-02 1.60349649604376361805568862450855e-03 1.34746115929350578771517277232306e-04
            1.11662181081691909423359021502620e-02 9.87230285395787054980587527097668e-01 1.34746115929350578771517277232306e-04
            1.60349649604376361805568862450855e-03 1.11662181081691909423359021502620e-02 1.34746115929350578771517277232306e-04
            1.60349649604376361805568862450855e-03 9.87230285395787054980587527097668e-01 1.34746115929350578771517277232306e-04
            9.87230285395787054980587527097668e-01 1.11662181081691909423359021502620e-02 1.34746115929350578771517277232306e-04
            9.87230285395787054980587527097668e-01 1.60349649604376361805568862450855e-03 1.34746115929350578771517277232306e-04
            2.08045649279087141447774911284796e-01 1.01545985226834029058196051664709e-01 4.69316533814183138584308352392327e-03
            2.08045649279087141447774911284796e-01 6.90408365494078801738453421421582e-01 4.69316533814183138584308352392327e-03
            1.01545985226834029058196051664709e-01 2.08045649279087141447774911284796e-01 4.69316533814183138584308352392327e-03
            1.01545985226834029058196051664709e-01 6.90408365494078801738453421421582e-01 4.69316533814183138584308352392327e-03
            6.90408365494078801738453421421582e-01 2.08045649279087141447774911284796e-01 4.69316533814183138584308352392327e-03
            6.90408365494078801738453421421582e-01 1.01545985226834029058196051664709e-01 4.69316533814183138584308352392327e-03
            3.92210114980434843001688705044216e-01 4.15270612688226833930649206649832e-02 3.91145922915323886182026313917959e-03
            3.92210114980434843001688705044216e-01 5.66262823750742438910776854754658e-01 3.91145922915323886182026313917959e-03
            4.15270612688226833930649206649832e-02 3.92210114980434843001688705044216e-01 3.91145922915323886182026313917959e-03
            4.15270612688226833930649206649832e-02 5.66262823750742438910776854754658e-01 3.91145922915323886182026313917959e-03
            5.66262823750742438910776854754658e-01 3.92210114980434843001688705044216e-01 3.91145922915323886182026313917959e-03
            5.66262823750742438910776854754658e-01 4.15270612688226833930649206649832e-02 3.91145922915323886182026313917959e-03
            3.59711275509975836328635523386765e-01 9.38490411451324418878883193428919e-02 5.31519818159830598203186013961385e-03
            3.59711275509975836328635523386765e-01 5.46439683344891680150112733826973e-01 5.31519818159830598203186013961385e-03
            9.38490411451324418878883193428919e-02 3.59711275509975836328635523386765e-01 5.31519818159830598203186013961385e-03
            9.38490411451324418878883193428919e-02 5.46439683344891680150112733826973e-01 5.31519818159830598203186013961385e-03
            5.46439683344891680150112733826973e-01 3.59711275509975836328635523386765e-01 5.31519818159830598203186013961385e-03
            5.46439683344891680150112733826973e-01 9.38490411451324418878883193428919e-02 5.31519818159830598203186013961385e-03
            2.45874699482875436196138707600767e-01 8.68602980438413878327974515514143e-03 1.52561941167259529535116246989901e-03
            2.45874699482875436196138707600767e-01 7.45439270712740409408070263452828e-01 1.52561941167259529535116246989901e-03
            8.68602980438413878327974515514143e-03 2.45874699482875436196138707600767e-01 1.52561941167259529535116246989901e-03
            8.68602980438413878327974515514143e-03 7.45439270712740409408070263452828e-01 1.52561941167259529535116246989901e-03
            7.45439270712740409408070263452828e-01 2.45874699482875436196138707600767e-01 1.52561941167259529535116246989901e-03
            7.45439270712740409408070263452828e-01 8.68602980438413878327974515514143e-03 1.52561941167259529535116246989901e-03
            1.67002738174923143210648390777351e-01 1.75891240440456259053192411556665e-02 1.91268333586500560369869639032459e-03
            1.67002738174923143210648390777351e-01 8.15408137781031161495093328994699e-01 1.91268333586500560369869639032459e-03
            1.75891240440456259053192411556665e-02 1.67002738174923143210648390777351e-01 1.91268333586500560369869639032459e-03
            1.75891240440456259053192411556665e-02 8.15408137781031161495093328994699e-01 1.91268333586500560369869639032459e-03
            8.15408137781031161495093328994699e-01 1.67002738174923143210648390777351e-01 1.91268333586500560369869639032459e-03
            8.15408137781031161495093328994699e-01 1.75891240440456259053192411556665e-02 1.91268333586500560369869639032459e-03
            1.15008598631946429291517119963828e-01 5.52352451221255549840272180972534e-03 8.70847615650874578435702844103616e-04
            1.15008598631946429291517119963828e-01 8.79467876855840935412800263293320e-01 8.70847615650874578435702844103616e-04
            5.52352451221255549840272180972534e-03 1.15008598631946429291517119963828e-01 8.70847615650874578435702844103616e-04
            5.52352451221255549840272180972534e-03 8.79467876855840935412800263293320e-01 8.70847615650874578435702844103616e-04
            8.79467876855840935412800263293320e-01 1.15008598631946429291517119963828e-01 8.70847615650874578435702844103616e-04
            8.79467876855840935412800263293320e-01 5.52352451221255549840272180972534e-03 8.70847615650874578435702844103616e-04
            3.15395398117319147868187201311230e-01 2.38589269426556054021215658167421e-02 2.83457674083272903944874521187103e-03
            3.15395398117319147868187201311230e-01 6.60745674940025229382456473103957e-01 2.83457674083272903944874521187103e-03
            2.38589269426556054021215658167421e-02 3.15395398117319147868187201311230e-01 2.83457674083272903944874521187103e-03
            2.38589269426556054021215658167421e-02 6.60745674940025229382456473103957e-01 2.83457674083272903944874521187103e-03
            6.60745674940025229382456473103957e-01 3.15395398117319147868187201311230e-01 2.83457674083272903944874521187103e-03
            6.60745674940025229382456473103957e-01 2.38589269426556054021215658167421e-02 2.83457674083272903944874521187103e-03
            2.23222650224820601261299657380732e-01 4.02953345447718061511643838912278e-02 3.29099888192610273360272366005574e-03
            2.23222650224820601261299657380732e-01 7.36482015230407571770854247006355e-01 3.29099888192610273360272366005574e-03
            4.02953345447718061511643838912278e-02 2.23222650224820601261299657380732e-01 3.29099888192610273360272366005574e-03
            4.02953345447718061511643838912278e-02 7.36482015230407571770854247006355e-01 3.29099888192610273360272366005574e-03
            7.36482015230407571770854247006355e-01 2.23222650224820601261299657380732e-01 3.29099888192610273360272366005574e-03
            7.36482015230407571770854247006355e-01 4.02953345447718061511643838912278e-02 3.29099888192610273360272366005574e-03
            2.88395859918732400117846736975480e-01 6.78784043114470936330917538725771e-02 4.58946208246379782486057408164015e-03
            2.88395859918732400117846736975480e-01 6.43725735769820506249061509151943e-01 4.58946208246379782486057408164015e-03
            6.78784043114470936330917538725771e-02 2.88395859918732400117846736975480e-01 4.58946208246379782486057408164015e-03
            6.78784043114470936330917538725771e-02 6.43725735769820506249061509151943e-01 4.58946208246379782486057408164015e-03
            6.43725735769820506249061509151943e-01 2.88395859918732400117846736975480e-01 4.58946208246379782486057408164015e-03
            6.43725735769820506249061509151943e-01 6.78784043114470936330917538725771e-02 4.58946208246379782486057408164015e-03
            2.67366350272775554763882155384636e-01 1.39535607181082654815185151164769e-01 6.25918324941721233856117834193356e-03
            2.67366350272775554763882155384636e-01 5.93098042546141845932083924708422e-01 6.25918324941721233856117834193356e-03
            1.39535607181082654815185151164769e-01 2.67366350272775554763882155384636e-01 6.25918324941721233856117834193356e-03
            1.39535607181082654815185151164769e-01 5.93098042546141845932083924708422e-01 6.25918324941721233856117834193356e-03
            5.93098042546141845932083924708422e-01 2.67366350272775554763882155384636e-01 6.25918324941721233856117834193356e-03
            5.93098042546141845932083924708422e-01 1.39535607181082654815185151164769e-01 6.25918324941721233856117834193356e-03
            4.05765395298891551334463656530716e-01 8.06658570416661502289201735038660e-03 1.82067020564036837620980335827880e-03
            4.05765395298891551334463656530716e-01 5.86168018996941819764856518304441e-01 1.82067020564036837620980335827880e-03
            8.06658570416661502289201735038660e-03 4.05765395298891551334463656530716e-01 1.82067020564036837620980335827880e-03
            8.06658570416661502289201735038660e-03 5.86168018996941819764856518304441e-01 1.82067020564036837620980335827880e-03
            5.86168018996941819764856518304441e-01 4.05765395298891551334463656530716e-01 1.82067020564036837620980335827880e-03
            5.86168018996941819764856518304441e-01 8.06658570416661502289201735038660e-03 1.82067020564036837620980335827880e-03
            1.86320727675359537611754490171734e-01 1.23446812287404965226972342051681e-04 3.44336312520880421985530750461635e-04
            1.86320727675359537611754490171734e-01 8.13555825512353059103531904838746e-01 3.44336312520880421985530750461635e-04
            1.23446812287404965226972342051681e-04 1.86320727675359537611754490171734e-01 3.44336312520880421985530750461635e-04
            1.23446812287404965226972342051681e-04 8.13555825512353059103531904838746e-01 3.44336312520880421985530750461635e-04
            8.13555825512353059103531904838746e-01 1.86320727675359537611754490171734e-01 3.44336312520880421985530750461635e-04
            8.13555825512353059103531904838746e-01 1.23446812287404965226972342051681e-04 3.44336312520880421985530750461635e-04
            ];
        
    case  30
        
        % ALG. DEG.:   30
        % PTS CARD.:  171
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.6e-15
        
        xyw=[
            3.31872493664435528476497339056550e-03 3.31872493664435528476497339056550e-03 8.58649506855248810001379755618700e-05
            3.31872493664435528476497339056550e-03 9.93362550126711263409617913566763e-01 8.58649506855248810001379755618700e-05
            9.93362550126711263409617913566763e-01 3.31872493664435528476497339056550e-03 8.58649506855248810001379755618700e-05
            7.23724072246778316053195112544927e-02 7.23724072246778316053195112544927e-02 1.90096633420776218656444989107968e-03
            7.23724072246778316053195112544927e-02 8.55255185550644281278209746233188e-01 1.90096633420776218656444989107968e-03
            8.55255185550644281278209746233188e-01 7.23724072246778316053195112544927e-02 1.90096633420776218656444989107968e-03
            4.71579102421717660797995108623581e-02 4.71579102421717660797995108623581e-02 1.42221683384375035923141705751505e-03
            4.71579102421717660797995108623581e-02 9.05684179515656495595976593904197e-01 1.42221683384375035923141705751505e-03
            9.05684179515656495595976593904197e-01 4.71579102421717660797995108623581e-02 1.42221683384375035923141705751505e-03
            4.68030173651125425138275204517413e-01 4.68030173651125425138275204517413e-01 3.89206436616543107603893858481570e-03
            4.68030173651125425138275204517413e-01 6.39396526977491497234495909651741e-02 3.89206436616543107603893858481570e-03
            6.39396526977491497234495909651741e-02 4.68030173651125425138275204517413e-01 3.89206436616543107603893858481570e-03
            1.26866046744675987217876311774489e-02 1.26866046744675987217876311774489e-02 4.23275072664445290923640463631727e-04
            1.26866046744675987217876311774489e-02 9.74626790651064833781447305227630e-01 4.23275072664445290923640463631727e-04
            9.74626790651064833781447305227630e-01 1.26866046744675987217876311774489e-02 4.23275072664445290923640463631727e-04
            1.21591508222727892607473165753618e-01 1.21591508222727892607473165753618e-01 3.69345591745715105119751697770880e-03
            1.21591508222727892607473165753618e-01 7.56816983554544187029478052863851e-01 3.69345591745715105119751697770880e-03
            7.56816983554544187029478052863851e-01 1.21591508222727892607473165753618e-01 3.69345591745715105119751697770880e-03
            1.82409561517452856183041376425535e-01 1.82409561517452856183041376425535e-01 5.37642548271641609880511225583177e-03
            1.82409561517452856183041376425535e-01 6.35180876965094287633917247148929e-01 5.37642548271641609880511225583177e-03
            6.35180876965094287633917247148929e-01 1.82409561517452856183041376425535e-01 5.37642548271641609880511225583177e-03
            3.62287279352949798116867441422073e-01 3.62287279352949798116867441422073e-01 7.79804566291291290786036327631336e-03
            3.62287279352949798116867441422073e-01 2.75425441294100403766265117155854e-01 7.79804566291291290786036327631336e-03
            2.75425441294100403766265117155854e-01 3.62287279352949798116867441422073e-01 7.79804566291291290786036327631336e-03
            4.36743248548460194502496278801118e-01 4.36743248548460194502496278801118e-01 6.20247951407301625209589346354733e-03
            4.36743248548460194502496278801118e-01 1.26513502903079610995007442397764e-01 6.20247951407301625209589346354733e-03
            1.26513502903079610995007442397764e-01 4.36743248548460194502496278801118e-01 6.20247951407301625209589346354733e-03
            2.72428040783928182566597797631402e-01 2.72428040783928182566597797631402e-01 7.45727538134289408278121413786721e-03
            2.72428040783928182566597797631402e-01 4.55143918432143634866804404737195e-01 7.45727538134289408278121413786721e-03
            4.55143918432143634866804404737195e-01 2.72428040783928182566597797631402e-01 7.45727538134289408278121413786721e-03
            4.97319339000308557885432492184918e-01 4.97319339000308557885432492184918e-01 1.39565905987038690883039748769079e-03
            4.97319339000308557885432492184918e-01 5.36132199938288422913501563016325e-03 1.39565905987038690883039748769079e-03
            5.36132199938288422913501563016325e-03 4.97319339000308557885432492184918e-01 1.39565905987038690883039748769079e-03
            2.59053884521067367963098604377592e-01 4.78351231407725743793868389275303e-02 2.10737923195622109395763743577845e-03
            2.59053884521067367963098604377592e-01 6.93110992338160092351984076231020e-01 2.10737923195622109395763743577845e-03
            4.78351231407725743793868389275303e-02 2.59053884521067367963098604377592e-01 2.10737923195622109395763743577845e-03
            4.78351231407725743793868389275303e-02 6.93110992338160092351984076231020e-01 2.10737923195622109395763743577845e-03
            6.93110992338160092351984076231020e-01 2.59053884521067367963098604377592e-01 2.10737923195622109395763743577845e-03
            6.93110992338160092351984076231020e-01 4.78351231407725743793868389275303e-02 2.10737923195622109395763743577845e-03
            3.91572188291256340075818798140972e-01 7.96595269316006521220785430159594e-02 2.96246137254601270283327352217384e-03
            3.91572188291256340075818798140972e-01 5.28768284777142993924314851028612e-01 2.96246137254601270283327352217384e-03
            7.96595269316006521220785430159594e-02 3.91572188291256340075818798140972e-01 2.96246137254601270283327352217384e-03
            7.96595269316006521220785430159594e-02 5.28768284777142993924314851028612e-01 2.96246137254601270283327352217384e-03
            5.28768284777142993924314851028612e-01 3.91572188291256340075818798140972e-01 2.96246137254601270283327352217384e-03
            5.28768284777142993924314851028612e-01 7.96595269316006521220785430159594e-02 2.96246137254601270283327352217384e-03
            3.56140283296233539633846021388308e-01 5.76934012738742479919906713803357e-02 2.97211009221224915083370277102404e-03
            3.56140283296233539633846021388308e-01 5.86166315429892281763102346303640e-01 2.97211009221224915083370277102404e-03
            5.76934012738742479919906713803357e-02 3.56140283296233539633846021388308e-01 2.97211009221224915083370277102404e-03
            5.76934012738742479919906713803357e-02 5.86166315429892281763102346303640e-01 2.97211009221224915083370277102404e-03
            5.86166315429892281763102346303640e-01 3.56140283296233539633846021388308e-01 2.97211009221224915083370277102404e-03
            5.86166315429892281763102346303640e-01 5.76934012738742479919906713803357e-02 2.97211009221224915083370277102404e-03
            2.83012497349588776351936303399270e-01 7.72614375768841260549990579420410e-02 3.43859219376796941547436148312045e-03
            2.83012497349588776351936303399270e-01 6.39726065073527139226428062102059e-01 3.43859219376796941547436148312045e-03
            7.72614375768841260549990579420410e-02 2.83012497349588776351936303399270e-01 3.43859219376796941547436148312045e-03
            7.72614375768841260549990579420410e-02 6.39726065073527139226428062102059e-01 3.43859219376796941547436148312045e-03
            6.39726065073527139226428062102059e-01 2.83012497349588776351936303399270e-01 3.43859219376796941547436148312045e-03
            6.39726065073527139226428062102059e-01 7.72614375768841260549990579420410e-02 3.43859219376796941547436148312045e-03
            2.41613762451524383134682238960522e-01 2.27583842950000725391390687946114e-02 2.04624948417360353822291862968541e-03
            2.41613762451524383134682238960522e-01 7.35627853253475461059451845358126e-01 2.04624948417360353822291862968541e-03
            2.27583842950000725391390687946114e-02 2.41613762451524383134682238960522e-01 2.04624948417360353822291862968541e-03
            2.27583842950000725391390687946114e-02 7.35627853253475461059451845358126e-01 2.04624948417360353822291862968541e-03
            7.35627853253475461059451845358126e-01 2.41613762451524383134682238960522e-01 2.04624948417360353822291862968541e-03
            7.35627853253475461059451845358126e-01 2.27583842950000725391390687946114e-02 2.04624948417360353822291862968541e-03
            2.52781247187932933684351155534387e-01 1.23811978770674652183814146155783e-01 4.43632725300868030471335146103229e-03
            2.52781247187932933684351155534387e-01 6.23406774041392441887410313938744e-01 4.43632725300868030471335146103229e-03
            1.23811978770674652183814146155783e-01 2.52781247187932933684351155534387e-01 4.43632725300868030471335146103229e-03
            1.23811978770674652183814146155783e-01 6.23406774041392441887410313938744e-01 4.43632725300868030471335146103229e-03
            6.23406774041392441887410313938744e-01 2.52781247187932933684351155534387e-01 4.43632725300868030471335146103229e-03
            6.23406774041392441887410313938744e-01 1.23811978770674652183814146155783e-01 4.43632725300868030471335146103229e-03
            1.84906106383917107072178964699560e-01 1.15881967236100599794035304057616e-01 3.76716146477324830693644841517198e-03
            1.84906106383917107072178964699560e-01 6.99211926379982307011573539057281e-01 3.76716146477324830693644841517198e-03
            1.15881967236100599794035304057616e-01 1.84906106383917107072178964699560e-01 3.76716146477324830693644841517198e-03
            1.15881967236100599794035304057616e-01 6.99211926379982307011573539057281e-01 3.76716146477324830693644841517198e-03
            6.99211926379982307011573539057281e-01 1.84906106383917107072178964699560e-01 3.76716146477324830693644841517198e-03
            6.99211926379982307011573539057281e-01 1.15881967236100599794035304057616e-01 3.76716146477324830693644841517198e-03
            1.93704377153640139308521384009509e-01 6.65174447818816239053063554820255e-02 3.40200337805094364662661376996766e-03
            1.93704377153640139308521384009509e-01 7.39778178064478209030596644879552e-01 3.40200337805094364662661376996766e-03
            6.65174447818816239053063554820255e-02 1.93704377153640139308521384009509e-01 3.40200337805094364662661376996766e-03
            6.65174447818816239053063554820255e-02 7.39778178064478209030596644879552e-01 3.40200337805094364662661376996766e-03
            7.39778178064478209030596644879552e-01 1.93704377153640139308521384009509e-01 3.40200337805094364662661376996766e-03
            7.39778178064478209030596644879552e-01 6.65174447818816239053063554820255e-02 3.40200337805094364662661376996766e-03
            7.64012484393875451749167382331507e-02 4.43878137706136145085311639491010e-03 6.03348284271060112185502966042350e-04
            7.64012484393875451749167382331507e-02 9.19159970183551111588826643128414e-01 6.03348284271060112185502966042350e-04
            4.43878137706136145085311639491010e-03 7.64012484393875451749167382331507e-02 6.03348284271060112185502966042350e-04
            4.43878137706136145085311639491010e-03 9.19159970183551111588826643128414e-01 6.03348284271060112185502966042350e-04
            9.19159970183551111588826643128414e-01 7.64012484393875451749167382331507e-02 6.03348284271060112185502966042350e-04
            9.19159970183551111588826643128414e-01 4.43878137706136145085311639491010e-03 6.03348284271060112185502966042350e-04
            2.09048087452689629328617115788802e-01 4.66357939268862656950753731166515e-03 9.79474288946623498561261378370091e-04
            2.09048087452689629328617115788802e-01 7.86288333154621765785918796609621e-01 9.79474288946623498561261378370091e-04
            4.66357939268862656950753731166515e-03 2.09048087452689629328617115788802e-01 9.79474288946623498561261378370091e-04
            4.66357939268862656950753731166515e-03 7.86288333154621765785918796609621e-01 9.79474288946623498561261378370091e-04
            7.86288333154621765785918796609621e-01 2.09048087452689629328617115788802e-01 9.79474288946623498561261378370091e-04
            7.86288333154621765785918796609621e-01 4.66357939268862656950753731166515e-03 9.79474288946623498561261378370091e-04
            2.98542994059242672300058529799571e-01 4.70368176447704539056005046404607e-03 1.13722295049934135971470006154505e-03
            2.98542994059242672300058529799571e-01 6.96753324176280330881638747086981e-01 1.13722295049934135971470006154505e-03
            4.70368176447704539056005046404607e-03 2.98542994059242672300058529799571e-01 1.13722295049934135971470006154505e-03
            4.70368176447704539056005046404607e-03 6.96753324176280330881638747086981e-01 1.13722295049934135971470006154505e-03
            6.96753324176280330881638747086981e-01 2.98542994059242672300058529799571e-01 1.13722295049934135971470006154505e-03
            6.96753324176280330881638747086981e-01 4.70368176447704539056005046404607e-03 1.13722295049934135971470006154505e-03
            3.34379040034031072003983808826888e-01 2.51820667038687127925467024169848e-02 2.68234242309316021912746208499811e-03
            3.34379040034031072003983808826888e-01 6.40438893262100239489598152431427e-01 2.68234242309316021912746208499811e-03
            2.51820667038687127925467024169848e-02 3.34379040034031072003983808826888e-01 2.68234242309316021912746208499811e-03
            2.51820667038687127925467024169848e-02 6.40438893262100239489598152431427e-01 2.68234242309316021912746208499811e-03
            6.40438893262100239489598152431427e-01 3.34379040034031072003983808826888e-01 2.68234242309316021912746208499811e-03
            6.40438893262100239489598152431427e-01 2.51820667038687127925467024169848e-02 2.68234242309316021912746208499811e-03
            1.23230802380695114717212845789618e-01 6.57765738247428899398983048740774e-02 2.93503490153276857047015546697821e-03
            1.23230802380695114717212845789618e-01 8.10992623794561939831737618078478e-01 2.93503490153276857047015546697821e-03
            6.57765738247428899398983048740774e-02 1.23230802380695114717212845789618e-01 2.93503490153276857047015546697821e-03
            6.57765738247428899398983048740774e-02 8.10992623794561939831737618078478e-01 2.93503490153276857047015546697821e-03
            8.10992623794561939831737618078478e-01 1.23230802380695114717212845789618e-01 2.93503490153276857047015546697821e-03
            8.10992623794561939831737618078478e-01 6.57765738247428899398983048740774e-02 2.93503490153276857047015546697821e-03
            3.38514162429845721913324041452142e-01 1.26124094984989471024405816024228e-01 5.65151177146875315082663604471236e-03
            3.38514162429845721913324041452142e-01 5.35361742585164779306694526894717e-01 5.65151177146875315082663604471236e-03
            1.26124094984989471024405816024228e-01 3.38514162429845721913324041452142e-01 5.65151177146875315082663604471236e-03
            1.26124094984989471024405816024228e-01 5.35361742585164779306694526894717e-01 5.65151177146875315082663604471236e-03
            5.35361742585164779306694526894717e-01 3.38514162429845721913324041452142e-01 5.65151177146875315082663604471236e-03
            5.35361742585164779306694526894717e-01 1.26124094984989471024405816024228e-01 5.65151177146875315082663604471236e-03
            3.54403602180687449241247577447211e-01 1.94870950923518471142870112089440e-01 7.15824263483380834188762875669454e-03
            3.54403602180687449241247577447211e-01 4.50725446895794079615882310463348e-01 7.15824263483380834188762875669454e-03
            1.94870950923518471142870112089440e-01 3.54403602180687449241247577447211e-01 7.15824263483380834188762875669454e-03
            1.94870950923518471142870112089440e-01 4.50725446895794079615882310463348e-01 7.15824263483380834188762875669454e-03
            4.50725446895794079615882310463348e-01 3.54403602180687449241247577447211e-01 7.15824263483380834188762875669454e-03
            4.50725446895794079615882310463348e-01 1.94870950923518471142870112089440e-01 7.15824263483380834188762875669454e-03
            2.63068297757808267078871722333133e-01 1.91001424572283140523865085924626e-01 6.46301172505916055738284242693226e-03
            2.63068297757808267078871722333133e-01 5.45930277669908647908414423000067e-01 6.46301172505916055738284242693226e-03
            1.91001424572283140523865085924626e-01 2.63068297757808267078871722333133e-01 6.46301172505916055738284242693226e-03
            1.91001424572283140523865085924626e-01 5.45930277669908647908414423000067e-01 6.46301172505916055738284242693226e-03
            5.45930277669908647908414423000067e-01 2.63068297757808267078871722333133e-01 6.46301172505916055738284242693226e-03
            5.45930277669908647908414423000067e-01 1.91001424572283140523865085924626e-01 6.46301172505916055738284242693226e-03
            4.34566173964669621998524462469504e-01 2.75334061245498980252666143542228e-02 3.14079829044583772046372871500353e-03
            4.34566173964669621998524462469504e-01 5.37900419910780525079019298573257e-01 3.14079829044583772046372871500353e-03
            2.75334061245498980252666143542228e-02 4.34566173964669621998524462469504e-01 3.14079829044583772046372871500353e-03
            2.75334061245498980252666143542228e-02 5.37900419910780525079019298573257e-01 3.14079829044583772046372871500353e-03
            5.37900419910780525079019298573257e-01 4.34566173964669621998524462469504e-01 3.14079829044583772046372871500353e-03
            5.37900419910780525079019298573257e-01 2.75334061245498980252666143542228e-02 3.14079829044583772046372871500353e-03
            1.64070987069878326813920921267709e-01 2.80639219813729780261812862818260e-02 2.35096849705298789959262428794773e-03
            1.64070987069878326813920921267709e-01 8.07865090948748698629344744404079e-01 2.35096849705298789959262428794773e-03
            2.80639219813729780261812862818260e-02 1.64070987069878326813920921267709e-01 2.35096849705298789959262428794773e-03
            2.80639219813729780261812862818260e-02 8.07865090948748698629344744404079e-01 2.35096849705298789959262428794773e-03
            8.07865090948748698629344744404079e-01 1.64070987069878326813920921267709e-01 2.35096849705298789959262428794773e-03
            8.07865090948748698629344744404079e-01 2.80639219813729780261812862818260e-02 2.35096849705298789959262428794773e-03
            4.26819997060804007249501523801882e-02 1.59024162689347069166156956043778e-02 9.15253438074415705261177578222487e-04
            4.26819997060804007249501523801882e-02 9.41415584024984819500048160989536e-01 9.15253438074415705261177578222487e-04
            1.59024162689347069166156956043778e-02 4.26819997060804007249501523801882e-02 9.15253438074415705261177578222487e-04
            1.59024162689347069166156956043778e-02 9.41415584024984819500048160989536e-01 9.15253438074415705261177578222487e-04
            9.41415584024984819500048160989536e-01 4.26819997060804007249501523801882e-02 9.15253438074415705261177578222487e-04
            9.41415584024984819500048160989536e-01 1.59024162689347069166156956043778e-02 9.15253438074415705261177578222487e-04
            9.39597946527298694130436729210487e-02 2.72942306520957758686485306043323e-02 1.91094027354751392830389811194891e-03
            9.39597946527298694130436729210487e-02 8.78745974695174347779413892567391e-01 1.91094027354751392830389811194891e-03
            2.72942306520957758686485306043323e-02 9.39597946527298694130436729210487e-02 1.91094027354751392830389811194891e-03
            2.72942306520957758686485306043323e-02 8.78745974695174347779413892567391e-01 1.91094027354751392830389811194891e-03
            8.78745974695174347779413892567391e-01 9.39597946527298694130436729210487e-02 1.91094027354751392830389811194891e-03
            8.78745974695174347779413892567391e-01 2.72942306520957758686485306043323e-02 1.91094027354751392830389811194891e-03
            1.35408853519934452291906268328603e-01 5.69121144541610401307307753882014e-03 9.59940278734460589703247901383065e-04
            1.35408853519934452291906268328603e-01 8.58899935034649453235999772005016e-01 9.59940278734460589703247901383065e-04
            5.69121144541610401307307753882014e-03 1.35408853519934452291906268328603e-01 9.59940278734460589703247901383065e-04
            5.69121144541610401307307753882014e-03 8.58899935034649453235999772005016e-01 9.59940278734460589703247901383065e-04
            8.58899935034649453235999772005016e-01 1.35408853519934452291906268328603e-01 9.59940278734460589703247901383065e-04
            8.58899935034649453235999772005016e-01 5.69121144541610401307307753882014e-03 9.59940278734460589703247901383065e-04
            3.96221514739659108617075844449573e-01 5.16234701662132353106038351597817e-03 1.32128396158162826103110720765699e-03
            3.96221514739659108617075844449573e-01 5.98616138243719619893568051338661e-01 1.32128396158162826103110720765699e-03
            5.16234701662132353106038351597817e-03 3.96221514739659108617075844449573e-01 1.32128396158162826103110720765699e-03
            5.16234701662132353106038351597817e-03 5.98616138243719619893568051338661e-01 1.32128396158162826103110720765699e-03
            5.98616138243719619893568051338661e-01 3.96221514739659108617075844449573e-01 1.32128396158162826103110720765699e-03
            5.98616138243719619893568051338661e-01 5.16234701662132353106038351597817e-03 1.32128396158162826103110720765699e-03
            2.94840425976739400892867593029223e-02 5.33708660694491229814495625305426e-04 1.67810855733201483557665012469329e-04
            2.94840425976739400892867593029223e-02 9.69982248741631636335114308167249e-01 1.67810855733201483557665012469329e-04
            5.33708660694491229814495625305426e-04 2.94840425976739400892867593029223e-02 1.67810855733201483557665012469329e-04
            5.33708660694491229814495625305426e-04 9.69982248741631636335114308167249e-01 1.67810855733201483557665012469329e-04
            9.69982248741631636335114308167249e-01 2.94840425976739400892867593029223e-02 1.67810855733201483557665012469329e-04
            9.69982248741631636335114308167249e-01 5.33708660694491229814495625305426e-04 1.67810855733201483557665012469329e-04
            ];
        
    case  31
        
        % ALG. DEG.:   31
        % PTS CARD.:  181
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.7e-15
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 4.17156040694318525074990589018853e-03
            4.86152467913059771476014248037245e-01 4.86152467913059771476014248037245e-01 2.02199419825968259092907608476253e-03
            4.86152467913059771476014248037245e-01 2.76950641738804570479715039255098e-02 2.02199419825968259092907608476253e-03
            2.76950641738804570479715039255098e-02 4.86152467913059771476014248037245e-01 2.02199419825968259092907608476253e-03
            4.19588080864260470903559507860336e-01 4.19588080864260470903559507860336e-01 4.42154132154415607097730500640864e-03
            4.19588080864260470903559507860336e-01 1.60823838271479058192880984279327e-01 4.42154132154415607097730500640864e-03
            1.60823838271479058192880984279327e-01 4.19588080864260470903559507860336e-01 4.42154132154415607097730500640864e-03
            7.92067965066985402167176744114840e-02 7.92067965066985402167176744114840e-02 2.11186492555161513945405360459517e-03
            7.92067965066985402167176744114840e-02 8.41586406986602919566564651177032e-01 2.11186492555161513945405360459517e-03
            8.41586406986602919566564651177032e-01 7.92067965066985402167176744114840e-02 2.11186492555161513945405360459517e-03
            1.77313492497139200965161620615618e-02 1.77313492497139200965161620615618e-02 4.93433072513541656593794026974820e-04
            1.77313492497139200965161620615618e-02 9.64537301500572152868073771969648e-01 4.93433072513541656593794026974820e-04
            9.64537301500572152868073771969648e-01 1.77313492497139200965161620615618e-02 4.93433072513541656593794026974820e-04
            2.18436496790928025291123049100861e-01 2.18436496790928025291123049100861e-01 5.61850175400658690544331008709378e-03
            2.18436496790928025291123049100861e-01 5.63127006418143949417753901798278e-01 5.61850175400658690544331008709378e-03
            5.63127006418143949417753901798278e-01 2.18436496790928025291123049100861e-01 5.61850175400658690544331008709378e-03
            1.72450787183886011799671678090817e-01 1.72450787183886011799671678090817e-01 4.77959798336420951803127366019908e-03
            1.72450787183886011799671678090817e-01 6.55098425632227976400656643818365e-01 4.77959798336420951803127366019908e-03
            6.55098425632227976400656643818365e-01 1.72450787183886011799671678090817e-01 4.77959798336420951803127366019908e-03
            5.10951808223812711795552843341284e-02 5.10951808223812711795552843341284e-02 1.62954332509288556118942636885549e-03
            5.10951808223812711795552843341284e-02 8.97809638355237416007526007888373e-01 1.62954332509288556118942636885549e-03
            8.97809638355237416007526007888373e-01 5.10951808223812711795552843341284e-02 1.62954332509288556118942636885549e-03
            4.97748620626470295036369861918502e-01 4.97748620626470295036369861918502e-01 1.16383032102821597567055000865821e-03
            4.97748620626470295036369861918502e-01 4.50275874705940992726027616299689e-03 1.16383032102821597567055000865821e-03
            4.50275874705940992726027616299689e-03 4.97748620626470295036369861918502e-01 1.16383032102821597567055000865821e-03
            3.85168469245680045176527528383303e-01 3.85168469245680045176527528383303e-01 6.86969759828413156776960590832459e-03
            3.85168469245680045176527528383303e-01 2.29663061508639909646944943233393e-01 6.86969759828413156776960590832459e-03
            2.29663061508639909646944943233393e-01 3.85168469245680045176527528383303e-01 6.86969759828413156776960590832459e-03
            2.97491490367992827970056168851443e-01 2.97491490367992827970056168851443e-01 6.17952025467345157638821007139995e-03
            2.97491490367992827970056168851443e-01 4.05017019264014344059887662297115e-01 6.17952025467345157638821007139995e-03
            4.05017019264014344059887662297115e-01 2.97491490367992827970056168851443e-01 6.17952025467345157638821007139995e-03
            5.08280946272773696825719724756709e-03 5.08280946272773696825719724756709e-03 1.72200689636290549646216452295278e-04
            5.08280946272773696825719724756709e-03 9.89834381074544578105189884809079e-01 1.72200689636290549646216452295278e-04
            9.89834381074544578105189884809079e-01 5.08280946272773696825719724756709e-03 1.72200689636290549646216452295278e-04
            4.50340941739018174594377796893241e-01 4.50340941739018174594377796893241e-01 5.21698864789029055633040599104788e-03
            4.50340941739018174594377796893241e-01 9.93181165219636508112444062135182e-02 5.21698864789029055633040599104788e-03
            9.93181165219636508112444062135182e-02 4.50340941739018174594377796893241e-01 5.21698864789029055633040599104788e-03
            1.39090628472094579581863627026905e-01 1.06669680864077562332248305665416e-01 2.90314662418183350167666390007071e-03
            1.39090628472094579581863627026905e-01 7.54239690663827899719251490751049e-01 2.90314662418183350167666390007071e-03
            1.06669680864077562332248305665416e-01 1.39090628472094579581863627026905e-01 2.90314662418183350167666390007071e-03
            1.06669680864077562332248305665416e-01 7.54239690663827899719251490751049e-01 2.90314662418183350167666390007071e-03
            7.54239690663827899719251490751049e-01 1.39090628472094579581863627026905e-01 2.90314662418183350167666390007071e-03
            7.54239690663827899719251490751049e-01 1.06669680864077562332248305665416e-01 2.90314662418183350167666390007071e-03
            1.97233130789653043901665796511224e-01 3.26115566372831879038418634308982e-03 6.41925849384848050752094827942074e-04
            1.97233130789653043901665796511224e-01 7.99505713546618590470416165771894e-01 6.41925849384848050752094827942074e-04
            3.26115566372831879038418634308982e-03 1.97233130789653043901665796511224e-01 6.41925849384848050752094827942074e-04
            3.26115566372831879038418634308982e-03 7.99505713546618590470416165771894e-01 6.41925849384848050752094827942074e-04
            7.99505713546618590470416165771894e-01 1.97233130789653043901665796511224e-01 6.41925849384848050752094827942074e-04
            7.99505713546618590470416165771894e-01 3.26115566372831879038418634308982e-03 6.41925849384848050752094827942074e-04
            2.38524150984310012724165517283836e-01 1.31398178404036562044732150411619e-02 1.41325713796386929002601995364330e-03
            2.38524150984310012724165517283836e-01 7.48336031175286353622766455373494e-01 1.41325713796386929002601995364330e-03
            1.31398178404036562044732150411619e-02 2.38524150984310012724165517283836e-01 1.41325713796386929002601995364330e-03
            1.31398178404036562044732150411619e-02 7.48336031175286353622766455373494e-01 1.41325713796386929002601995364330e-03
            7.48336031175286353622766455373494e-01 2.38524150984310012724165517283836e-01 1.41325713796386929002601995364330e-03
            7.48336031175286353622766455373494e-01 1.31398178404036562044732150411619e-02 1.41325713796386929002601995364330e-03
            2.97507293859899553389869453212668e-02 2.34422112868689367162766679086872e-03 2.52754523436129129813215721256370e-04
            2.97507293859899553389869453212668e-02 9.67905049485323076829956789879361e-01 2.52754523436129129813215721256370e-04
            2.34422112868689367162766679086872e-03 2.97507293859899553389869453212668e-02 2.52754523436129129813215721256370e-04
            2.34422112868689367162766679086872e-03 9.67905049485323076829956789879361e-01 2.52754523436129129813215721256370e-04
            9.67905049485323076829956789879361e-01 2.97507293859899553389869453212668e-02 2.52754523436129129813215721256370e-04
            9.67905049485323076829956789879361e-01 2.34422112868689367162766679086872e-03 2.52754523436129129813215721256370e-04
            3.95783933296885381825092053986737e-01 1.96140036989502072536928345414253e-03 6.51347724486725748564708293741887e-04
            3.95783933296885381825092053986737e-01 6.02254666333219668139520308614010e-01 6.51347724486725748564708293741887e-04
            1.96140036989502072536928345414253e-03 3.95783933296885381825092053986737e-01 6.51347724486725748564708293741887e-04
            1.96140036989502072536928345414253e-03 6.02254666333219668139520308614010e-01 6.51347724486725748564708293741887e-04
            6.02254666333219668139520308614010e-01 3.95783933296885381825092053986737e-01 6.51347724486725748564708293741887e-04
            6.02254666333219668139520308614010e-01 1.96140036989502072536928345414253e-03 6.51347724486725748564708293741887e-04
            3.51680058776080939342278952608467e-01 1.52636565775219751328606321294501e-01 5.09628972217063790084434771188171e-03
            3.51680058776080939342278952608467e-01 4.95683375448699337084690341725945e-01 5.09628972217063790084434771188171e-03
            1.52636565775219751328606321294501e-01 3.51680058776080939342278952608467e-01 5.09628972217063790084434771188171e-03
            1.52636565775219751328606321294501e-01 4.95683375448699337084690341725945e-01 5.09628972217063790084434771188171e-03
            4.95683375448699337084690341725945e-01 3.51680058776080939342278952608467e-01 5.09628972217063790084434771188171e-03
            4.95683375448699337084690341725945e-01 1.52636565775219751328606321294501e-01 5.09628972217063790084434771188171e-03
            2.02482194261758341458090626474586e-01 1.13425278347502905829102815005172e-01 3.93343524786906890283733062574356e-03
            2.02482194261758341458090626474586e-01 6.84092527390738780468382174149156e-01 3.93343524786906890283733062574356e-03
            1.13425278347502905829102815005172e-01 2.02482194261758341458090626474586e-01 3.93343524786906890283733062574356e-03
            1.13425278347502905829102815005172e-01 6.84092527390738780468382174149156e-01 3.93343524786906890283733062574356e-03
            6.84092527390738780468382174149156e-01 2.02482194261758341458090626474586e-01 3.93343524786906890283733062574356e-03
            6.84092527390738780468382174149156e-01 1.13425278347502905829102815005172e-01 3.93343524786906890283733062574356e-03
            4.22553955222215060771873140765820e-01 5.45213844446831366608030577936006e-02 3.56040462070939586064488580063880e-03
            4.22553955222215060771873140765820e-01 5.22924660333101809506217705347808e-01 3.56040462070939586064488580063880e-03
            5.45213844446831366608030577936006e-02 4.22553955222215060771873140765820e-01 3.56040462070939586064488580063880e-03
            5.45213844446831366608030577936006e-02 5.22924660333101809506217705347808e-01 3.56040462070939586064488580063880e-03
            5.22924660333101809506217705347808e-01 4.22553955222215060771873140765820e-01 3.56040462070939586064488580063880e-03
            5.22924660333101809506217705347808e-01 5.45213844446831366608030577936006e-02 3.56040462070939586064488580063880e-03
            1.24680508847443971554014296998503e-01 6.06049291607722168118144168147410e-02 2.48205723825383481676021268924615e-03
            1.24680508847443971554014296998503e-01 8.14714561991783825511959094001213e-01 2.48205723825383481676021268924615e-03
            6.06049291607722168118144168147410e-02 1.24680508847443971554014296998503e-01 2.48205723825383481676021268924615e-03
            6.06049291607722168118144168147410e-02 8.14714561991783825511959094001213e-01 2.48205723825383481676021268924615e-03
            8.14714561991783825511959094001213e-01 1.24680508847443971554014296998503e-01 2.48205723825383481676021268924615e-03
            8.14714561991783825511959094001213e-01 6.06049291607722168118144168147410e-02 2.48205723825383481676021268924615e-03
            2.55546236402857285252565588962170e-01 3.75889521575177576462323258965625e-02 2.71821873650677668307618084497790e-03
            2.55546236402857285252565588962170e-01 7.06864811439624873834475238254527e-01 2.71821873650677668307618084497790e-03
            3.75889521575177576462323258965625e-02 2.55546236402857285252565588962170e-01 2.71821873650677668307618084497790e-03
            3.75889521575177576462323258965625e-02 7.06864811439624873834475238254527e-01 2.71821873650677668307618084497790e-03
            7.06864811439624873834475238254527e-01 2.55546236402857285252565588962170e-01 2.71821873650677668307618084497790e-03
            7.06864811439624873834475238254527e-01 3.75889521575177576462323258965625e-02 2.71821873650677668307618084497790e-03
            7.45031299353863113443807719704637e-02 4.38633017006165564638520137918931e-03 5.76291470042130803304314401458441e-04
            7.45031299353863113443807719704637e-02 9.21110539894552049489107048430014e-01 5.76291470042130803304314401458441e-04
            4.38633017006165564638520137918931e-03 7.45031299353863113443807719704637e-02 5.76291470042130803304314401458441e-04
            4.38633017006165564638520137918931e-03 9.21110539894552049489107048430014e-01 5.76291470042130803304314401458441e-04
            9.21110539894552049489107048430014e-01 7.45031299353863113443807719704637e-02 5.76291470042130803304314401458441e-04
            9.21110539894552049489107048430014e-01 4.38633017006165564638520137918931e-03 5.76291470042130803304314401458441e-04
            4.16807275523650544446496724049211e-01 1.72734143249371197015307188848965e-02 2.08916634591076524896102739603521e-03
            4.16807275523650544446496724049211e-01 5.65919310151412391363123788323719e-01 2.08916634591076524896102739603521e-03
            1.72734143249371197015307188848965e-02 4.16807275523650544446496724049211e-01 2.08916634591076524896102739603521e-03
            1.72734143249371197015307188848965e-02 5.65919310151412391363123788323719e-01 2.08916634591076524896102739603521e-03
            5.65919310151412391363123788323719e-01 4.16807275523650544446496724049211e-01 2.08916634591076524896102739603521e-03
            5.65919310151412391363123788323719e-01 1.72734143249371197015307188848965e-02 2.08916634591076524896102739603521e-03
            1.92871723006198453331805353627715e-01 6.25780220143109106034629007808689e-02 3.18121152643343870283776375629259e-03
            1.92871723006198453331805353627715e-01 7.44550254979490566675792706519132e-01 3.18121152643343870283776375629259e-03
            6.25780220143109106034629007808689e-02 1.92871723006198453331805353627715e-01 3.18121152643343870283776375629259e-03
            6.25780220143109106034629007808689e-02 7.44550254979490566675792706519132e-01 3.18121152643343870283776375629259e-03
            7.44550254979490566675792706519132e-01 1.92871723006198453331805353627715e-01 3.18121152643343870283776375629259e-03
            7.44550254979490566675792706519132e-01 6.25780220143109106034629007808689e-02 3.18121152643343870283776375629259e-03
            4.73478189395524634575096456501342e-02 1.84079470312829319988967569088345e-02 9.06384644141955365601037719613942e-04
            4.73478189395524634575096456501342e-02 9.34244234029164677401979588466929e-01 9.06384644141955365601037719613942e-04
            1.84079470312829319988967569088345e-02 4.73478189395524634575096456501342e-02 9.06384644141955365601037719613942e-04
            1.84079470312829319988967569088345e-02 9.34244234029164677401979588466929e-01 9.06384644141955365601037719613942e-04
            9.34244234029164677401979588466929e-01 4.73478189395524634575096456501342e-02 9.06384644141955365601037719613942e-04
            9.34244234029164677401979588466929e-01 1.84079470312829319988967569088345e-02 9.06384644141955365601037719613942e-04
            3.41905005347527657377781906689052e-01 4.29032089427714424800797132775187e-02 3.34497962252281290720934414650856e-03
            3.41905005347527657377781906689052e-01 6.15191785709700900142138380033430e-01 3.34497962252281290720934414650856e-03
            4.29032089427714424800797132775187e-02 3.41905005347527657377781906689052e-01 3.34497962252281290720934414650856e-03
            4.29032089427714424800797132775187e-02 6.15191785709700900142138380033430e-01 3.34497962252281290720934414650856e-03
            6.15191785709700900142138380033430e-01 3.41905005347527657377781906689052e-01 3.34497962252281290720934414650856e-03
            6.15191785709700900142138380033430e-01 4.29032089427714424800797132775187e-02 3.34497962252281290720934414650856e-03
            3.25813953057509653188361653519678e-01 1.31890837790157033904137051649741e-02 1.81985674247372021354596238751355e-03
            3.25813953057509653188361653519678e-01 6.60996963163474560154497794428607e-01 1.81985674247372021354596238751355e-03
            1.31890837790157033904137051649741e-02 3.25813953057509653188361653519678e-01 1.81985674247372021354596238751355e-03
            1.31890837790157033904137051649741e-02 6.60996963163474560154497794428607e-01 1.81985674247372021354596238751355e-03
            6.60996963163474560154497794428607e-01 3.25813953057509653188361653519678e-01 1.81985674247372021354596238751355e-03
            6.60996963163474560154497794428607e-01 1.31890837790157033904137051649741e-02 1.81985674247372021354596238751355e-03
            2.89425948757137285749507782384171e-01 1.19688163070959881391364110214681e-03 4.64685168846559022107906011811451e-04
            2.89425948757137285749507782384171e-01 7.09377169612153024580436522228410e-01 4.64685168846559022107906011811451e-04
            1.19688163070959881391364110214681e-03 2.89425948757137285749507782384171e-01 4.64685168846559022107906011811451e-04
            1.19688163070959881391364110214681e-03 7.09377169612153024580436522228410e-01 4.64685168846559022107906011811451e-04
            7.09377169612153024580436522228410e-01 2.89425948757137285749507782384171e-01 4.64685168846559022107906011811451e-04
            7.09377169612153024580436522228410e-01 1.19688163070959881391364110214681e-03 4.64685168846559022107906011811451e-04
            2.98789795676013270053772430401295e-01 2.19132761430176847827766550835804e-01 6.44779339399209703437287899419061e-03
            2.98789795676013270053772430401295e-01 4.82077442893809882118461018762900e-01 6.44779339399209703437287899419061e-03
            2.19132761430176847827766550835804e-01 2.98789795676013270053772430401295e-01 6.44779339399209703437287899419061e-03
            2.19132761430176847827766550835804e-01 4.82077442893809882118461018762900e-01 6.44779339399209703437287899419061e-03
            4.82077442893809882118461018762900e-01 2.98789795676013270053772430401295e-01 6.44779339399209703437287899419061e-03
            4.82077442893809882118461018762900e-01 2.19132761430176847827766550835804e-01 6.44779339399209703437287899419061e-03
            2.74475747742794984329606222672737e-01 8.11298915693309669050492516362283e-02 4.23794885856830939180328599036329e-03
            2.74475747742794984329606222672737e-01 6.44394360687874034887556717876578e-01 4.23794885856830939180328599036329e-03
            8.11298915693309669050492516362283e-02 2.74475747742794984329606222672737e-01 4.23794885856830939180328599036329e-03
            8.11298915693309669050492516362283e-02 6.44394360687874034887556717876578e-01 4.23794885856830939180328599036329e-03
            6.44394360687874034887556717876578e-01 2.74475747742794984329606222672737e-01 4.23794885856830939180328599036329e-03
            6.44394360687874034887556717876578e-01 8.11298915693309669050492516362283e-02 4.23794885856830939180328599036329e-03
            3.53229873891415580011710062535712e-01 9.52781144820168263143145281901525e-02 4.81113776584836221050789362152500e-03
            3.53229873891415580011710062535712e-01 5.51492011626567579796187601459678e-01 4.81113776584836221050789362152500e-03
            9.52781144820168263143145281901525e-02 3.53229873891415580011710062535712e-01 4.81113776584836221050789362152500e-03
            9.52781144820168263143145281901525e-02 5.51492011626567579796187601459678e-01 4.81113776584836221050789362152500e-03
            5.51492011626567579796187601459678e-01 3.53229873891415580011710062535712e-01 4.81113776584836221050789362152500e-03
            5.51492011626567579796187601459678e-01 9.52781144820168263143145281901525e-02 4.81113776584836221050789362152500e-03
            2.66367882475071815484568560350453e-01 1.46888708002111900707831182444352e-01 5.79983451569281074217121485503412e-03
            2.66367882475071815484568560350453e-01 5.86743409522816339318751488463022e-01 5.79983451569281074217121485503412e-03
            1.46888708002111900707831182444352e-01 2.66367882475071815484568560350453e-01 5.79983451569281074217121485503412e-03
            1.46888708002111900707831182444352e-01 5.86743409522816339318751488463022e-01 5.79983451569281074217121485503412e-03
            5.86743409522816339318751488463022e-01 2.66367882475071815484568560350453e-01 5.79983451569281074217121485503412e-03
            5.86743409522816339318751488463022e-01 1.46888708002111900707831182444352e-01 5.79983451569281074217121485503412e-03
            9.60481521010510957969685819080041e-02 2.75306300785874945302733607377377e-02 1.83329335246342297639776663942257e-03
            9.60481521010510957969685819080041e-02 8.76421217820361420081098913215101e-01 1.83329335246342297639776663942257e-03
            2.75306300785874945302733607377377e-02 9.60481521010510957969685819080041e-02 1.83329335246342297639776663942257e-03
            2.75306300785874945302733607377377e-02 8.76421217820361420081098913215101e-01 1.83329335246342297639776663942257e-03
            8.76421217820361420081098913215101e-01 9.60481521010510957969685819080041e-02 1.83329335246342297639776663942257e-03
            8.76421217820361420081098913215101e-01 2.75306300785874945302733607377377e-02 1.83329335246342297639776663942257e-03
            1.67407824778356056905082027697063e-01 2.58751082199381692017237099889826e-02 2.22266899562620053390493701783726e-03
            1.67407824778356056905082027697063e-01 8.06717067001705756545959502545884e-01 2.22266899562620053390493701783726e-03
            2.58751082199381692017237099889826e-02 1.67407824778356056905082027697063e-01 2.22266899562620053390493701783726e-03
            2.58751082199381692017237099889826e-02 8.06717067001705756545959502545884e-01 2.22266899562620053390493701783726e-03
            8.06717067001705756545959502545884e-01 1.67407824778356056905082027697063e-01 2.22266899562620053390493701783726e-03
            8.06717067001705756545959502545884e-01 2.58751082199381692017237099889826e-02 2.22266899562620053390493701783726e-03
            1.30677633770288320613772725664603e-01 5.79731868473983870393428929901347e-03 9.10626392061236224206854927842869e-04
            1.30677633770288320613772725664603e-01 8.63525047544971835478122557105962e-01 9.10626392061236224206854927842869e-04
            5.79731868473983870393428929901347e-03 1.30677633770288320613772725664603e-01 9.10626392061236224206854927842869e-04
            5.79731868473983870393428929901347e-03 8.63525047544971835478122557105962e-01 9.10626392061236224206854927842869e-04
            8.63525047544971835478122557105962e-01 1.30677633770288320613772725664603e-01 9.10626392061236224206854927842869e-04
            8.63525047544971835478122557105962e-01 5.79731868473983870393428929901347e-03 9.10626392061236224206854927842869e-04
            ];
        
    case  32
        
        % ALG. DEG.:   32
        % PTS CARD.:  193
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.7e-15
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 6.51503549788128803049724879770110e-03
            1.45744546236985564845178853232710e-03 1.45744546236985564845178853232710e-03 2.66619511667080602026168312246313e-05
            1.45744546236985564845178853232710e-03 9.97085109075260267452733842219459e-01 2.66619511667080602026168312246313e-05
            9.97085109075260267452733842219459e-01 1.45744546236985564845178853232710e-03 2.66619511667080602026168312246313e-05
            4.98650893578847342890725258257589e-01 4.98650893578847342890725258257589e-01 7.07313623831419785274732259239272e-04
            4.98650893578847342890725258257589e-01 2.69821284230531421854948348482139e-03 7.07313623831419785274732259239272e-04
            2.69821284230531421854948348482139e-03 4.98650893578847342890725258257589e-01 7.07313623831419785274732259239272e-04
            4.81168115995836753739922642125748e-01 4.81168115995836753739922642125748e-01 2.79000757120663510704727805489256e-03
            4.81168115995836753739922642125748e-01 3.76637680083264925201547157485038e-02 2.79000757120663510704727805489256e-03
            3.76637680083264925201547157485038e-02 4.81168115995836753739922642125748e-01 2.79000757120663510704727805489256e-03
            4.52285756511600522955518499657046e-01 4.52285756511600522955518499657046e-01 4.18929535206312257666905196629159e-03
            4.52285756511600522955518499657046e-01 9.54284869767989540889630006859079e-02 4.18929535206312257666905196629159e-03
            9.54284869767989540889630006859079e-02 4.52285756511600522955518499657046e-01 4.18929535206312257666905196629159e-03
            3.76328758047918987550417568854755e-01 3.76328758047918987550417568854755e-01 6.33064728024301630326942103010879e-03
            3.76328758047918987550417568854755e-01 2.47342483904162024899164862290490e-01 6.33064728024301630326942103010879e-03
            2.47342483904162024899164862290490e-01 3.76328758047918987550417568854755e-01 6.33064728024301630326942103010879e-03
            2.83018279849573173301280348823639e-01 2.83018279849573173301280348823639e-01 6.44136658858768165053509946460508e-03
            2.83018279849573173301280348823639e-01 4.33963440300853653397439302352723e-01 6.44136658858768165053509946460508e-03
            4.33963440300853653397439302352723e-01 2.83018279849573173301280348823639e-01 6.44136658858768165053509946460508e-03
            4.13679784325413357759515520228888e-01 4.13679784325413357759515520228888e-01 5.44023301346712503251401571446877e-03
            4.13679784325413357759515520228888e-01 1.72640431349173284480968959542224e-01 5.44023301346712503251401571446877e-03
            1.72640431349173284480968959542224e-01 4.13679784325413357759515520228888e-01 5.44023301346712503251401571446877e-03
            3.02691910404441627546745507970627e-02 3.02691910404441627546745507970627e-02 9.29639126596945065961197229853497e-04
            3.02691910404441627546745507970627e-02 9.39461617919111646735075282776961e-01 9.29639126596945065961197229853497e-04
            9.39461617919111646735075282776961e-01 3.02691910404441627546745507970627e-02 9.29639126596945065961197229853497e-04
            1.28101964801118073378205508561223e-02 1.28101964801118073378205508561223e-02 4.00870525836536913209112720224425e-04
            1.28101964801118073378205508561223e-02 9.74379607039776329813207667029928e-01 4.00870525836536913209112720224425e-04
            9.74379607039776329813207667029928e-01 1.28101964801118073378205508561223e-02 4.00870525836536913209112720224425e-04
            6.20106498262642247265574724224280e-02 6.20106498262642247265574724224280e-02 1.98817429746268046000801099637556e-03
            6.20106498262642247265574724224280e-02 8.75978700347471495035733823897317e-01 1.98817429746268046000801099637556e-03
            8.75978700347471495035733823897317e-01 6.20106498262642247265574724224280e-02 1.98817429746268046000801099637556e-03
            2.29277996928069882631717746335198e-01 2.29277996928069882631717746335198e-01 6.26779886282444187450835215713596e-03
            2.29277996928069882631717746335198e-01 5.41444006143860234736564507329604e-01 6.26779886282444187450835215713596e-03
            5.41444006143860234736564507329604e-01 2.29277996928069882631717746335198e-01 6.26779886282444187450835215713596e-03
            1.01958826900269583504154979891609e-01 1.01958826900269583504154979891609e-01 3.31469955752957713690509677917362e-03
            1.01958826900269583504154979891609e-01 7.96082346199460832991690040216781e-01 3.31469955752957713690509677917362e-03
            7.96082346199460832991690040216781e-01 1.01958826900269583504154979891609e-01 3.31469955752957713690509677917362e-03
            2.06831686315902013584278051894216e-01 1.58867941229046738538244198934990e-01 4.17460956476896957040301217034539e-03
            2.06831686315902013584278051894216e-01 6.34300372455051331144204596057534e-01 4.17460956476896957040301217034539e-03
            1.58867941229046738538244198934990e-01 2.06831686315902013584278051894216e-01 4.17460956476896957040301217034539e-03
            1.58867941229046738538244198934990e-01 6.34300372455051331144204596057534e-01 4.17460956476896957040301217034539e-03
            6.34300372455051331144204596057534e-01 2.06831686315902013584278051894216e-01 4.17460956476896957040301217034539e-03
            6.34300372455051331144204596057534e-01 1.58867941229046738538244198934990e-01 4.17460956476896957040301217034539e-03
            4.46761426533167027574933172218152e-01 1.47664424308546586572798986480848e-02 1.52032612888649935889429087865210e-03
            4.46761426533167027574933172218152e-01 5.38472131035978351931703400623519e-01 1.52032612888649935889429087865210e-03
            1.47664424308546586572798986480848e-02 4.46761426533167027574933172218152e-01 1.52032612888649935889429087865210e-03
            1.47664424308546586572798986480848e-02 5.38472131035978351931703400623519e-01 1.52032612888649935889429087865210e-03
            5.38472131035978351931703400623519e-01 4.46761426533167027574933172218152e-01 1.52032612888649935889429087865210e-03
            5.38472131035978351931703400623519e-01 1.47664424308546586572798986480848e-02 1.52032612888649935889429087865210e-03
            2.32466207646378680840015817921085e-01 5.14311272402330696418282229842589e-02 2.64302180126624059602891136933067e-03
            2.32466207646378680840015817921085e-01 7.16102665113388270334837670816341e-01 2.64302180126624059602891136933067e-03
            5.14311272402330696418282229842589e-02 2.32466207646378680840015817921085e-01 2.64302180126624059602891136933067e-03
            5.14311272402330696418282229842589e-02 7.16102665113388270334837670816341e-01 2.64302180126624059602891136933067e-03
            7.16102665113388270334837670816341e-01 2.32466207646378680840015817921085e-01 2.64302180126624059602891136933067e-03
            7.16102665113388270334837670816341e-01 5.14311272402330696418282229842589e-02 2.64302180126624059602891136933067e-03
            1.16834926706957942244002879306208e-01 1.95597403645923935411321537003460e-02 1.24663496233453469606844432604476e-03
            1.16834926706957942244002879306208e-01 8.63605332928449653806524111132603e-01 1.24663496233453469606844432604476e-03
            1.95597403645923935411321537003460e-02 1.16834926706957942244002879306208e-01 1.24663496233453469606844432604476e-03
            1.95597403645923935411321537003460e-02 8.63605332928449653806524111132603e-01 1.24663496233453469606844432604476e-03
            8.63605332928449653806524111132603e-01 1.16834926706957942244002879306208e-01 1.24663496233453469606844432604476e-03
            8.63605332928449653806524111132603e-01 1.95597403645923935411321537003460e-02 1.24663496233453469606844432604476e-03
            3.28240545844934006325388509139884e-01 1.26185667501572066490078682932108e-02 1.51029052835681344875429221730201e-03
            3.28240545844934006325388509139884e-01 6.59140887404908837332584425894311e-01 1.51029052835681344875429221730201e-03
            1.26185667501572066490078682932108e-02 3.28240545844934006325388509139884e-01 1.51029052835681344875429221730201e-03
            1.26185667501572066490078682932108e-02 6.59140887404908837332584425894311e-01 1.51029052835681344875429221730201e-03
            6.59140887404908837332584425894311e-01 3.28240545844934006325388509139884e-01 1.51029052835681344875429221730201e-03
            6.59140887404908837332584425894311e-01 1.26185667501572066490078682932108e-02 1.51029052835681344875429221730201e-03
            2.59350932003120149271779837363283e-01 2.10621566756392201991143764416847e-02 1.82208778843809108584361755589498e-03
            2.59350932003120149271779837363283e-01 7.19586911321240529915144179540221e-01 1.82208778843809108584361755589498e-03
            2.10621566756392201991143764416847e-02 2.59350932003120149271779837363283e-01 1.82208778843809108584361755589498e-03
            2.10621566756392201991143764416847e-02 7.19586911321240529915144179540221e-01 1.82208778843809108584361755589498e-03
            7.19586911321240529915144179540221e-01 2.59350932003120149271779837363283e-01 1.82208778843809108584361755589498e-03
            7.19586911321240529915144179540221e-01 2.10621566756392201991143764416847e-02 1.82208778843809108584361755589498e-03
            1.65463998237349529940587444798439e-01 1.17543509254347977588395224302076e-01 3.74054484958007832373638734679844e-03
            1.65463998237349529940587444798439e-01 7.16992492508302436959866099641658e-01 3.74054484958007832373638734679844e-03
            1.17543509254347977588395224302076e-01 1.65463998237349529940587444798439e-01 3.74054484958007832373638734679844e-03
            1.17543509254347977588395224302076e-01 7.16992492508302436959866099641658e-01 3.74054484958007832373638734679844e-03
            7.16992492508302436959866099641658e-01 1.65463998237349529940587444798439e-01 3.74054484958007832373638734679844e-03
            7.16992492508302436959866099641658e-01 1.17543509254347977588395224302076e-01 3.74054484958007832373638734679844e-03
            3.82464907380937246816898777979077e-01 2.92988505404852273006355289908242e-02 2.28407706778600751856833639408251e-03
            3.82464907380937246816898777979077e-01 5.88236242078577609149192539916839e-01 2.28407706778600751856833639408251e-03
            2.92988505404852273006355289908242e-02 3.82464907380937246816898777979077e-01 2.28407706778600751856833639408251e-03
            2.92988505404852273006355289908242e-02 5.88236242078577609149192539916839e-01 2.28407706778600751856833639408251e-03
            5.88236242078577609149192539916839e-01 3.82464907380937246816898777979077e-01 2.28407706778600751856833639408251e-03
            5.88236242078577609149192539916839e-01 2.92988505404852273006355289908242e-02 2.28407706778600751856833639408251e-03
            3.97817072646720837347800170391565e-01 3.44282165227435561077884429437290e-03 8.42601820267144035678241742459704e-04
            3.97817072646720837347800170391565e-01 5.98740105701004776683760155719938e-01 8.42601820267144035678241742459704e-04
            3.44282165227435561077884429437290e-03 3.97817072646720837347800170391565e-01 8.42601820267144035678241742459704e-04
            3.44282165227435561077884429437290e-03 5.98740105701004776683760155719938e-01 8.42601820267144035678241742459704e-04
            5.98740105701004776683760155719938e-01 3.97817072646720837347800170391565e-01 8.42601820267144035678241742459704e-04
            5.98740105701004776683760155719938e-01 3.44282165227435561077884429437290e-03 8.42601820267144035678241742459704e-04
            3.25417469646690105111730417775107e-01 9.03702811448642562286437396323890e-02 4.10460332323965386425168588857559e-03
            3.25417469646690105111730417775107e-01 5.84212249208445721926352689479245e-01 4.10460332323965386425168588857559e-03
            9.03702811448642562286437396323890e-02 3.25417469646690105111730417775107e-01 4.10460332323965386425168588857559e-03
            9.03702811448642562286437396323890e-02 5.84212249208445721926352689479245e-01 4.10460332323965386425168588857559e-03
            5.84212249208445721926352689479245e-01 3.25417469646690105111730417775107e-01 4.10460332323965386425168588857559e-03
            5.84212249208445721926352689479245e-01 9.03702811448642562286437396323890e-02 4.10460332323965386425168588857559e-03
            1.13339191487603074826040483458200e-01 5.06976359267404738662143870442378e-02 2.23283842380996673179782696649909e-03
            1.13339191487603074826040483458200e-01 8.35963172585656444368851225590333e-01 2.23283842380996673179782696649909e-03
            5.06976359267404738662143870442378e-02 1.13339191487603074826040483458200e-01 2.23283842380996673179782696649909e-03
            5.06976359267404738662143870442378e-02 8.35963172585656444368851225590333e-01 2.23283842380996673179782696649909e-03
            8.35963172585656444368851225590333e-01 1.13339191487603074826040483458200e-01 2.23283842380996673179782696649909e-03
            8.35963172585656444368851225590333e-01 5.06976359267404738662143870442378e-02 2.23283842380996673179782696649909e-03
            3.11041632827641745251412430661730e-01 4.79444789155617986931012808327068e-02 3.10141736076142158010093830000642e-03
            3.11041632827641745251412430661730e-01 6.41013888256796504627743615856161e-01 3.10141736076142158010093830000642e-03
            4.79444789155617986931012808327068e-02 3.11041632827641745251412430661730e-01 3.10141736076142158010093830000642e-03
            4.79444789155617986931012808327068e-02 6.41013888256796504627743615856161e-01 3.10141736076142158010093830000642e-03
            6.41013888256796504627743615856161e-01 3.11041632827641745251412430661730e-01 3.10141736076142158010093830000642e-03
            6.41013888256796504627743615856161e-01 4.79444789155617986931012808327068e-02 3.10141736076142158010093830000642e-03
            2.98837135388912988887000210525002e-01 1.27665738027984911225065633999520e-03 4.44446271575963465678971164862787e-04
            2.98837135388912988887000210525002e-01 6.99886207230807189105803445272613e-01 4.44446271575963465678971164862787e-04
            1.27665738027984911225065633999520e-03 2.98837135388912988887000210525002e-01 4.44446271575963465678971164862787e-04
            1.27665738027984911225065633999520e-03 6.99886207230807189105803445272613e-01 4.44446271575963465678971164862787e-04
            6.99886207230807189105803445272613e-01 2.98837135388912988887000210525002e-01 4.44446271575963465678971164862787e-04
            6.99886207230807189105803445272613e-01 1.27665738027984911225065633999520e-03 4.44446271575963465678971164862787e-04
            1.52112280943783789721734933664266e-01 4.26256341511160007085035061891176e-03 7.42614664592583821264848431553673e-04
            1.52112280943783789721734933664266e-01 8.43625155641104629289372951461701e-01 7.42614664592583821264848431553673e-04
            4.26256341511160007085035061891176e-03 1.52112280943783789721734933664266e-01 7.42614664592583821264848431553673e-04
            4.26256341511160007085035061891176e-03 8.43625155641104629289372951461701e-01 7.42614664592583821264848431553673e-04
            8.43625155641104629289372951461701e-01 1.52112280943783789721734933664266e-01 7.42614664592583821264848431553673e-04
            8.43625155641104629289372951461701e-01 4.26256341511160007085035061891176e-03 7.42614664592583821264848431553673e-04
            8.86617050339115198021033847908257e-02 3.85251198871905528239856231209615e-03 5.56353172795318300786204801511303e-04
            8.86617050339115198021033847908257e-02 9.07485782977369370705389428621856e-01 5.56353172795318300786204801511303e-04
            3.85251198871905528239856231209615e-03 8.86617050339115198021033847908257e-02 5.56353172795318300786204801511303e-04
            3.85251198871905528239856231209615e-03 9.07485782977369370705389428621856e-01 5.56353172795318300786204801511303e-04
            9.07485782977369370705389428621856e-01 8.86617050339115198021033847908257e-02 5.56353172795318300786204801511303e-04
            9.07485782977369370705389428621856e-01 3.85251198871905528239856231209615e-03 5.56353172795318300786204801511303e-04
            3.19152580593060508640235184429912e-01 2.01900148632476683641101544708363e-01 5.98667955124138843198133841383424e-03
            3.19152580593060508640235184429912e-01 4.78947270774462807718663270861725e-01 5.98667955124138843198133841383424e-03
            2.01900148632476683641101544708363e-01 3.19152580593060508640235184429912e-01 5.98667955124138843198133841383424e-03
            2.01900148632476683641101544708363e-01 4.78947270774462807718663270861725e-01 5.98667955124138843198133841383424e-03
            4.78947270774462807718663270861725e-01 3.19152580593060508640235184429912e-01 5.98667955124138843198133841383424e-03
            4.78947270774462807718663270861725e-01 2.01900148632476683641101544708363e-01 5.98667955124138843198133841383424e-03
            1.66755693582133324159855192192481e-01 6.77103932212691633774070965046121e-02 2.94115803140992861791636059365374e-03
            1.66755693582133324159855192192481e-01 7.65533913196597581851676750375191e-01 2.94115803140992861791636059365374e-03
            6.77103932212691633774070965046121e-02 1.66755693582133324159855192192481e-01 2.94115803140992861791636059365374e-03
            6.77103932212691633774070965046121e-02 7.65533913196597581851676750375191e-01 2.94115803140992861791636059365374e-03
            7.65533913196597581851676750375191e-01 1.66755693582133324159855192192481e-01 2.94115803140992861791636059365374e-03
            7.65533913196597581851676750375191e-01 6.77103932212691633774070965046121e-02 2.94115803140992861791636059365374e-03
            2.42137924019432793087958089017775e-01 9.55724171106545339338111944016418e-02 4.22437173597556281379183928947896e-03
            2.42137924019432793087958089017775e-01 6.62289658869912645222655100951670e-01 4.22437173597556281379183928947896e-03
            9.55724171106545339338111944016418e-02 2.42137924019432793087958089017775e-01 4.22437173597556281379183928947896e-03
            9.55724171106545339338111944016418e-02 6.62289658869912645222655100951670e-01 4.22437173597556281379183928947896e-03
            6.62289658869912645222655100951670e-01 2.42137924019432793087958089017775e-01 4.22437173597556281379183928947896e-03
            6.62289658869912645222655100951670e-01 9.55724171106545339338111944016418e-02 4.22437173597556281379183928947896e-03
            2.79861747997829701795069468062138e-01 1.50370864573458273838468812755309e-01 5.10572795818304073223181660523551e-03
            2.79861747997829701795069468062138e-01 5.69767387428712024366461719182553e-01 5.10572795818304073223181660523551e-03
            1.50370864573458273838468812755309e-01 2.79861747997829701795069468062138e-01 5.10572795818304073223181660523551e-03
            1.50370864573458273838468812755309e-01 5.69767387428712024366461719182553e-01 5.10572795818304073223181660523551e-03
            5.69767387428712024366461719182553e-01 2.79861747997829701795069468062138e-01 5.10572795818304073223181660523551e-03
            5.69767387428712024366461719182553e-01 1.50370864573458273838468812755309e-01 5.10572795818304073223181660523551e-03
            6.73984002072566534824460404706770e-02 2.43498506845871223736033073237195e-02 1.24497957660826331395542698743384e-03
            6.73984002072566534824460404706770e-02 9.08251749108156203327268940483918e-01 1.24497957660826331395542698743384e-03
            2.43498506845871223736033073237195e-02 6.73984002072566534824460404706770e-02 1.24497957660826331395542698743384e-03
            2.43498506845871223736033073237195e-02 9.08251749108156203327268940483918e-01 1.24497957660826331395542698743384e-03
            9.08251749108156203327268940483918e-01 6.73984002072566534824460404706770e-02 1.24497957660826331395542698743384e-03
            9.08251749108156203327268940483918e-01 2.43498506845871223736033073237195e-02 1.24497957660826331395542698743384e-03
            3.71687966693420357433552680959110e-01 1.29387867366923425826996663090540e-01 4.94425282751114174989437444196483e-03
            3.71687966693420357433552680959110e-01 4.98924165939656272250601887208177e-01 4.94425282751114174989437444196483e-03
            1.29387867366923425826996663090540e-01 3.71687966693420357433552680959110e-01 4.94425282751114174989437444196483e-03
            1.29387867366923425826996663090540e-01 4.98924165939656272250601887208177e-01 4.94425282751114174989437444196483e-03
            4.98924165939656272250601887208177e-01 3.71687966693420357433552680959110e-01 4.94425282751114174989437444196483e-03
            4.98924165939656272250601887208177e-01 1.29387867366923425826996663090540e-01 4.94425282751114174989437444196483e-03
            4.05628149065882848134378946269862e-01 6.32274204565012321044292775695794e-02 3.70455657146824719749833576543097e-03
            4.05628149065882848134378946269862e-01 5.31144430477615947516767391789472e-01 3.70455657146824719749833576543097e-03
            6.32274204565012321044292775695794e-02 4.05628149065882848134378946269862e-01 3.70455657146824719749833576543097e-03
            6.32274204565012321044292775695794e-02 5.31144430477615947516767391789472e-01 3.70455657146824719749833576543097e-03
            5.31144430477615947516767391789472e-01 4.05628149065882848134378946269862e-01 3.70455657146824719749833576543097e-03
            5.31144430477615947516767391789472e-01 6.32274204565012321044292775695794e-02 3.70455657146824719749833576543097e-03
            2.21980165737481027754540718888165e-01 5.26663302076577251320532724321311e-03 9.72796620491533643552972687729152e-04
            2.21980165737481027754540718888165e-01 7.72753201241753173711401814216515e-01 9.72796620491533643552972687729152e-04
            5.26663302076577251320532724321311e-03 2.21980165737481027754540718888165e-01 9.72796620491533643552972687729152e-04
            5.26663302076577251320532724321311e-03 7.72753201241753173711401814216515e-01 9.72796620491533643552972687729152e-04
            7.72753201241753173711401814216515e-01 2.21980165737481027754540718888165e-01 9.72796620491533643552972687729152e-04
            7.72753201241753173711401814216515e-01 5.26663302076577251320532724321311e-03 9.72796620491533643552972687729152e-04
            1.78325682037862948137174612384115e-01 2.61588819796510496651364974241005e-02 2.13041829825008242546569015019031e-03
            1.78325682037862948137174612384115e-01 7.95515435982486085464415737078525e-01 2.13041829825008242546569015019031e-03
            2.61588819796510496651364974241005e-02 1.78325682037862948137174612384115e-01 2.13041829825008242546569015019031e-03
            2.61588819796510496651364974241005e-02 7.95515435982486085464415737078525e-01 2.13041829825008242546569015019031e-03
            7.95515435982486085464415737078525e-01 1.78325682037862948137174612384115e-01 2.13041829825008242546569015019031e-03
            7.95515435982486085464415737078525e-01 2.61588819796510496651364974241005e-02 2.13041829825008242546569015019031e-03
            4.26221905914095089418225370536675e-02 5.55476787043224142642605656305932e-03 5.27687184808808547147040979297117e-04
            4.26221905914095089418225370536675e-02 9.51823041538158332031116515281610e-01 5.27687184808808547147040979297117e-04
            5.55476787043224142642605656305932e-03 4.26221905914095089418225370536675e-02 5.27687184808808547147040979297117e-04
            5.55476787043224142642605656305932e-03 9.51823041538158332031116515281610e-01 5.27687184808808547147040979297117e-04
            9.51823041538158332031116515281610e-01 4.26221905914095089418225370536675e-02 5.27687184808808547147040979297117e-04
            9.51823041538158332031116515281610e-01 5.55476787043224142642605656305932e-03 5.27687184808808547147040979297117e-04
            1.43443003792803787377341251385587e-02 6.23098020345069585589070637610121e-04 8.50441238712439719375432711778728e-05
            1.43443003792803787377341251385587e-02 9.85032601600374535522064434189815e-01 8.50441238712439719375432711778728e-05
            6.23098020345069585589070637610121e-04 1.43443003792803787377341251385587e-02 8.50441238712439719375432711778728e-05
            6.23098020345069585589070637610121e-04 9.85032601600374535522064434189815e-01 8.50441238712439719375432711778728e-05
            9.85032601600374535522064434189815e-01 1.43443003792803787377341251385587e-02 8.50441238712439719375432711778728e-05
            9.85032601600374535522064434189815e-01 6.23098020345069585589070637610121e-04 8.50441238712439719375432711778728e-05
            ];
        
    case  33
        
        % ALG. DEG.:   33
        % PTS CARD.:  204
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.5e-15
        
        xyw=[
            8.66842399345696401624650206940714e-02 8.66842399345696401624650206940714e-02 1.60118200620710366376131528198812e-03
            8.66842399345696401624650206940714e-02 8.26631520130860719675069958611857e-01 1.60118200620710366376131528198812e-03
            8.26631520130860719675069958611857e-01 8.66842399345696401624650206940714e-02 1.60118200620710366376131528198812e-03
            4.09427960802643320370464152802015e-01 4.09427960802643320370464152802015e-01 3.41733996723404516784872164691933e-03
            4.09427960802643320370464152802015e-01 1.81144078394713359259071694395971e-01 3.41733996723404516784872164691933e-03
            1.81144078394713359259071694395971e-01 4.09427960802643320370464152802015e-01 3.41733996723404516784872164691933e-03
            3.78333538296242721798989805392921e-01 3.78333538296242721798989805392921e-01 4.46708090406837312474896961589366e-03
            3.78333538296242721798989805392921e-01 2.43332923407514556402020389214158e-01 4.46708090406837312474896961589366e-03
            2.43332923407514556402020389214158e-01 3.78333538296242721798989805392921e-01 4.46708090406837312474896961589366e-03
            4.98979717860051186217873464556760e-01 4.98979717860051186217873464556760e-01 5.20545618643210739745486037577393e-04
            4.98979717860051186217873464556760e-01 2.04056427989762756425307088647969e-03 5.20545618643210739745486037577393e-04
            2.04056427989762756425307088647969e-03 4.98979717860051186217873464556760e-01 5.20545618643210739745486037577393e-04
            2.59660668008721118305004438298056e-01 2.59660668008721118305004438298056e-01 4.86942861834268724691154872630250e-03
            2.59660668008721118305004438298056e-01 4.80678663982557763389991123403888e-01 4.86942861834268724691154872630250e-03
            4.80678663982557763389991123403888e-01 2.59660668008721118305004438298056e-01 4.86942861834268724691154872630250e-03
            4.78511973256382172614564751711441e-01 4.78511973256382172614564751711441e-01 2.73216284202301923703104513663220e-03
            4.78511973256382172614564751711441e-01 4.29760534872356547708704965771176e-02 2.73216284202301923703104513663220e-03
            4.29760534872356547708704965771176e-02 4.78511973256382172614564751711441e-01 2.73216284202301923703104513663220e-03
            2.35174430579130483431882225886511e-02 2.35174430579130483431882225886511e-02 6.90684531513952428352642876063783e-04
            2.35174430579130483431882225886511e-02 9.52965113884173931069199170451611e-01 6.90684531513952428352642876063783e-04
            9.52965113884173931069199170451611e-01 2.35174430579130483431882225886511e-02 6.90684531513952428352642876063783e-04
            3.06023861760360560424487630371004e-01 3.06023861760360560424487630371004e-01 6.04436999045828392274071205747532e-03
            3.06023861760360560424487630371004e-01 3.87952276479278879151024739257991e-01 6.04436999045828392274071205747532e-03
            3.87952276479278879151024739257991e-01 3.06023861760360560424487630371004e-01 6.04436999045828392274071205747532e-03
            1.46489050716642699168090757666505e-01 1.46489050716642699168090757666505e-01 3.68872324836709694281799976067759e-03
            1.46489050716642699168090757666505e-01 7.07021898566714601663818484666990e-01 3.68872324836709694281799976067759e-03
            7.07021898566714601663818484666990e-01 1.46489050716642699168090757666505e-01 3.68872324836709694281799976067759e-03
            5.43792032794962812425332288057689e-02 5.43792032794962812425332288057689e-02 1.60467522197081072406199542257355e-03
            5.43792032794962812425332288057689e-02 8.91241593441007395881570118945092e-01 1.60467522197081072406199542257355e-03
            8.91241593441007395881570118945092e-01 5.43792032794962812425332288057689e-02 1.60467522197081072406199542257355e-03
            2.18192015863253874208282923063962e-01 2.18192015863253874208282923063962e-01 4.96318808977210326749407443003292e-03
            2.18192015863253874208282923063962e-01 5.63615968273492251583434153872076e-01 4.96318808977210326749407443003292e-03
            5.63615968273492251583434153872076e-01 2.18192015863253874208282923063962e-01 4.96318808977210326749407443003292e-03
            4.91991344456553236774709603196243e-01 4.91991344456553236774709603196243e-01 1.93118101498302146418195590626965e-03
            4.91991344456553236774709603196243e-01 1.60173110868935264505807936075144e-02 1.93118101498302146418195590626965e-03
            1.60173110868935264505807936075144e-02 4.91991344456553236774709603196243e-01 1.93118101498302146418195590626965e-03
            4.53399961617805269398218115384225e-01 4.53399961617805269398218115384225e-01 4.66689888482329107849411187203259e-03
            4.53399961617805269398218115384225e-01 9.32000767643894612035637692315504e-02 4.66689888482329107849411187203259e-03
            9.32000767643894612035637692315504e-02 4.53399961617805269398218115384225e-01 4.66689888482329107849411187203259e-03
            4.72666051056800425728132708513840e-03 4.72666051056800425728132708513840e-03 1.46741366789008789326895221272196e-04
            4.72666051056800425728132708513840e-03 9.90546678978863948117350446409546e-01 1.46741366789008789326895221272196e-04
            9.90546678978863948117350446409546e-01 4.72666051056800425728132708513840e-03 1.46741366789008789326895221272196e-04
            2.58152928789506741491521779607865e-01 8.69286359632870375691027220454998e-02 2.50541172828744936745826166202278e-03
            2.58152928789506741491521779607865e-01 6.54918435247206165428224267088808e-01 2.50541172828744936745826166202278e-03
            8.69286359632870375691027220454998e-02 2.58152928789506741491521779607865e-01 2.50541172828744936745826166202278e-03
            8.69286359632870375691027220454998e-02 6.54918435247206165428224267088808e-01 2.50541172828744936745826166202278e-03
            6.54918435247206165428224267088808e-01 2.58152928789506741491521779607865e-01 2.50541172828744936745826166202278e-03
            6.54918435247206165428224267088808e-01 8.69286359632870375691027220454998e-02 2.50541172828744936745826166202278e-03
            2.84962401972082890466708704479970e-01 1.14092035322831353294503742290544e-01 3.13333575729897928149880748094347e-03
            2.84962401972082890466708704479970e-01 6.00945562705085700727636321971659e-01 3.13333575729897928149880748094347e-03
            1.14092035322831353294503742290544e-01 2.84962401972082890466708704479970e-01 3.13333575729897928149880748094347e-03
            1.14092035322831353294503742290544e-01 6.00945562705085700727636321971659e-01 3.13333575729897928149880748094347e-03
            6.00945562705085700727636321971659e-01 2.84962401972082890466708704479970e-01 3.13333575729897928149880748094347e-03
            6.00945562705085700727636321971659e-01 1.14092035322831353294503742290544e-01 3.13333575729897928149880748094347e-03
            1.34067633841934125138450895065034e-01 9.16653314400512203574677982942376e-02 2.55304615093165715097867085603411e-03
            1.34067633841934125138450895065034e-01 7.74267034718014723893020345713012e-01 2.55304615093165715097867085603411e-03
            9.16653314400512203574677982942376e-02 1.34067633841934125138450895065034e-01 2.55304615093165715097867085603411e-03
            9.16653314400512203574677982942376e-02 7.74267034718014723893020345713012e-01 2.55304615093165715097867085603411e-03
            7.74267034718014723893020345713012e-01 1.34067633841934125138450895065034e-01 2.55304615093165715097867085603411e-03
            7.74267034718014723893020345713012e-01 9.16653314400512203574677982942376e-02 2.55304615093165715097867085603411e-03
            3.26913750209671560842394910650910e-01 2.15939391243415657051230027718702e-01 4.81962342942039278365085053223993e-03
            3.26913750209671560842394910650910e-01 4.57146858546912837617526292888215e-01 4.81962342942039278365085053223993e-03
            2.15939391243415657051230027718702e-01 3.26913750209671560842394910650910e-01 4.81962342942039278365085053223993e-03
            2.15939391243415657051230027718702e-01 4.57146858546912837617526292888215e-01 4.81962342942039278365085053223993e-03
            4.57146858546912837617526292888215e-01 3.26913750209671560842394910650910e-01 4.81962342942039278365085053223993e-03
            4.57146858546912837617526292888215e-01 2.15939391243415657051230027718702e-01 4.81962342942039278365085053223993e-03
            1.00845715982775571695206906497333e-01 2.23903162865835385353374675787563e-02 1.21061901943424784881819800119729e-03
            1.00845715982775571695206906497333e-01 8.76763967730640914055584289599210e-01 1.21061901943424784881819800119729e-03
            2.23903162865835385353374675787563e-02 1.00845715982775571695206906497333e-01 1.21061901943424784881819800119729e-03
            2.23903162865835385353374675787563e-02 8.76763967730640914055584289599210e-01 1.21061901943424784881819800119729e-03
            8.76763967730640914055584289599210e-01 1.00845715982775571695206906497333e-01 1.21061901943424784881819800119729e-03
            8.76763967730640914055584289599210e-01 2.23903162865835385353374675787563e-02 1.21061901943424784881819800119729e-03
            3.55232825404763974397326364851324e-01 9.92419378880282920807331947798957e-02 3.84867336352746934949320412044926e-03
            3.55232825404763974397326364851324e-01 5.45525236707207761277516055997694e-01 3.84867336352746934949320412044926e-03
            9.92419378880282920807331947798957e-02 3.55232825404763974397326364851324e-01 3.84867336352746934949320412044926e-03
            9.92419378880282920807331947798957e-02 5.45525236707207761277516055997694e-01 3.84867336352746934949320412044926e-03
            5.45525236707207761277516055997694e-01 3.55232825404763974397326364851324e-01 3.84867336352746934949320412044926e-03
            5.45525236707207761277516055997694e-01 9.92419378880282920807331947798957e-02 3.84867336352746934949320412044926e-03
            1.58402734195724786969705633055128e-01 2.04436751441748401636466780928458e-02 1.49412234597451668537126945324189e-03
            1.58402734195724786969705633055128e-01 8.21153590660100296538814745872514e-01 1.49412234597451668537126945324189e-03
            2.04436751441748401636466780928458e-02 1.58402734195724786969705633055128e-01 1.49412234597451668537126945324189e-03
            2.04436751441748401636466780928458e-02 8.21153590660100296538814745872514e-01 1.49412234597451668537126945324189e-03
            8.21153590660100296538814745872514e-01 1.58402734195724786969705633055128e-01 1.49412234597451668537126945324189e-03
            8.21153590660100296538814745872514e-01 2.04436751441748401636466780928458e-02 1.49412234597451668537126945324189e-03
            4.01239748458054656943261306878412e-01 5.86121184566509023650127119253739e-02 3.09942701297168240678892026096491e-03
            4.01239748458054656943261306878412e-01 5.40148133085294412936150365567300e-01 3.09942701297168240678892026096491e-03
            5.86121184566509023650127119253739e-02 4.01239748458054656943261306878412e-01 3.09942701297168240678892026096491e-03
            5.86121184566509023650127119253739e-02 5.40148133085294412936150365567300e-01 3.09942701297168240678892026096491e-03
            5.40148133085294412936150365567300e-01 4.01239748458054656943261306878412e-01 3.09942701297168240678892026096491e-03
            5.40148133085294412936150365567300e-01 5.86121184566509023650127119253739e-02 3.09942701297168240678892026096491e-03
            2.27322013164966585385684538778150e-01 1.94249101655772464514182473749315e-02 1.65849864695653739093961220874007e-03
            2.27322013164966585385684538778150e-01 7.53253076669456178571238069707761e-01 1.65849864695653739093961220874007e-03
            1.94249101655772464514182473749315e-02 2.27322013164966585385684538778150e-01 1.65849864695653739093961220874007e-03
            1.94249101655772464514182473749315e-02 7.53253076669456178571238069707761e-01 1.65849864695653739093961220874007e-03
            7.53253076669456178571238069707761e-01 2.27322013164966585385684538778150e-01 1.65849864695653739093961220874007e-03
            7.53253076669456178571238069707761e-01 1.94249101655772464514182473749315e-02 1.65849864695653739093961220874007e-03
            1.04004000131427021269558963467716e-01 5.08701921937649848004525665601250e-02 1.85446480629463316888716484243105e-03
            1.04004000131427021269558963467716e-01 8.45125807674807938418837238714332e-01 1.85446480629463316888716484243105e-03
            5.08701921937649848004525665601250e-02 1.04004000131427021269558963467716e-01 1.85446480629463316888716484243105e-03
            5.08701921937649848004525665601250e-02 8.45125807674807938418837238714332e-01 1.85446480629463316888716484243105e-03
            8.45125807674807938418837238714332e-01 1.04004000131427021269558963467716e-01 1.85446480629463316888716484243105e-03
            8.45125807674807938418837238714332e-01 5.08701921937649848004525665601250e-02 1.85446480629463316888716484243105e-03
            2.51619019363036655612120284786215e-01 3.80625212881914110410463969458306e-03 7.71323259235281869297673473795385e-04
            2.51619019363036655612120284786215e-01 7.44574728508144212391073324397439e-01 7.71323259235281869297673473795385e-04
            3.80625212881914110410463969458306e-03 2.51619019363036655612120284786215e-01 7.71323259235281869297673473795385e-04
            3.80625212881914110410463969458306e-03 7.44574728508144212391073324397439e-01 7.71323259235281869297673473795385e-04
            7.44574728508144212391073324397439e-01 2.51619019363036655612120284786215e-01 7.71323259235281869297673473795385e-04
            7.44574728508144212391073324397439e-01 3.80625212881914110410463969458306e-03 7.71323259235281869297673473795385e-04
            1.94671527372046238024694275736692e-01 9.39537103695899794564283524778148e-02 3.30608620553421222662549006088284e-03
            1.94671527372046238024694275736692e-01 7.11374762258363713129938332713209e-01 3.30608620553421222662549006088284e-03
            9.39537103695899794564283524778148e-02 1.94671527372046238024694275736692e-01 3.30608620553421222662549006088284e-03
            9.39537103695899794564283524778148e-02 7.11374762258363713129938332713209e-01 3.30608620553421222662549006088284e-03
            7.11374762258363713129938332713209e-01 1.94671527372046238024694275736692e-01 3.30608620553421222662549006088284e-03
            7.11374762258363713129938332713209e-01 9.39537103695899794564283524778148e-02 3.30608620553421222662549006088284e-03
            1.75639631458033129707274611064349e-01 3.66697762401211810323609086026408e-03 6.77589097875325174769045588618610e-04
            1.75639631458033129707274611064349e-01 8.20693390917954790353405769565143e-01 6.77589097875325174769045588618610e-04
            3.66697762401211810323609086026408e-03 1.75639631458033129707274611064349e-01 6.77589097875325174769045588618610e-04
            3.66697762401211810323609086026408e-03 8.20693390917954790353405769565143e-01 6.77589097875325174769045588618610e-04
            8.20693390917954790353405769565143e-01 1.75639631458033129707274611064349e-01 6.77589097875325174769045588618610e-04
            8.20693390917954790353405769565143e-01 3.66697762401211810323609086026408e-03 6.77589097875325174769045588618610e-04
            1.65422503950626270663448735831480e-01 5.04257035060658373448738700517424e-02 2.45181417177645858546108925679619e-03
            1.65422503950626270663448735831480e-01 7.84151792543307912808359105838463e-01 2.45181417177645858546108925679619e-03
            5.04257035060658373448738700517424e-02 1.65422503950626270663448735831480e-01 2.45181417177645858546108925679619e-03
            5.04257035060658373448738700517424e-02 7.84151792543307912808359105838463e-01 2.45181417177645858546108925679619e-03
            7.84151792543307912808359105838463e-01 1.65422503950626270663448735831480e-01 2.45181417177645858546108925679619e-03
            7.84151792543307912808359105838463e-01 5.04257035060658373448738700517424e-02 2.45181417177645858546108925679619e-03
            6.04829240909602813824896827554767e-02 4.11214828094120989770976493105081e-03 4.57883078590504025238289154131621e-04
            6.04829240909602813824896827554767e-02 9.35404927628098570302483949490124e-01 4.57883078590504025238289154131621e-04
            4.11214828094120989770976493105081e-03 6.04829240909602813824896827554767e-02 4.57883078590504025238289154131621e-04
            4.11214828094120989770976493105081e-03 9.35404927628098570302483949490124e-01 4.57883078590504025238289154131621e-04
            9.35404927628098570302483949490124e-01 6.04829240909602813824896827554767e-02 4.57883078590504025238289154131621e-04
            9.35404927628098570302483949490124e-01 4.11214828094120989770976493105081e-03 4.57883078590504025238289154131621e-04
            3.32103565357523278223084162164014e-01 4.33667086499291924012178256475636e-03 9.10908132213261809845272765073787e-04
            3.32103565357523278223084162164014e-01 6.63559763777483824220837504981318e-01 9.10908132213261809845272765073787e-04
            4.33667086499291924012178256475636e-03 3.32103565357523278223084162164014e-01 9.10908132213261809845272765073787e-04
            4.33667086499291924012178256475636e-03 6.63559763777483824220837504981318e-01 9.10908132213261809845272765073787e-04
            6.63559763777483824220837504981318e-01 3.32103565357523278223084162164014e-01 9.10908132213261809845272765073787e-04
            6.63559763777483824220837504981318e-01 4.33667086499291924012178256475636e-03 9.10908132213261809845272765073787e-04
            3.79413216845638723295763838905259e-01 1.43764176240543023466145200472965e-01 4.62118995085480672307953753374932e-03
            3.79413216845638723295763838905259e-01 4.76822606913818169971364113735035e-01 4.62118995085480672307953753374932e-03
            1.43764176240543023466145200472965e-01 3.79413216845638723295763838905259e-01 4.62118995085480672307953753374932e-03
            1.43764176240543023466145200472965e-01 4.76822606913818169971364113735035e-01 4.62118995085480672307953753374932e-03
            4.76822606913818169971364113735035e-01 3.79413216845638723295763838905259e-01 4.62118995085480672307953753374932e-03
            4.76822606913818169971364113735035e-01 1.43764176240543023466145200472965e-01 4.62118995085480672307953753374932e-03
            5.65688972489456487480197210970800e-02 2.18658180283708390712060776195358e-02 9.80315616846607479351050074001250e-04
            5.65688972489456487480197210970800e-02 9.21565284722683508711327249329770e-01 9.80315616846607479351050074001250e-04
            2.18658180283708390712060776195358e-02 5.65688972489456487480197210970800e-02 9.80315616846607479351050074001250e-04
            2.18658180283708390712060776195358e-02 9.21565284722683508711327249329770e-01 9.80315616846607479351050074001250e-04
            9.21565284722683508711327249329770e-01 5.65688972489456487480197210970800e-02 9.80315616846607479351050074001250e-04
            9.21565284722683508711327249329770e-01 2.18658180283708390712060776195358e-02 9.80315616846607479351050074001250e-04
            2.90984916269330096127987417276017e-01 1.68307331342502364268653991530300e-01 5.10495889319825032343080906116484e-03
            2.90984916269330096127987417276017e-01 5.40707752388167595114509822451510e-01 5.10495889319825032343080906116484e-03
            1.68307331342502364268653991530300e-01 2.90984916269330096127987417276017e-01 5.10495889319825032343080906116484e-03
            1.68307331342502364268653991530300e-01 5.40707752388167595114509822451510e-01 5.10495889319825032343080906116484e-03
            5.40707752388167595114509822451510e-01 2.90984916269330096127987417276017e-01 5.10495889319825032343080906116484e-03
            5.40707752388167595114509822451510e-01 1.68307331342502364268653991530300e-01 5.10495889319825032343080906116484e-03
            2.36845394483151533426834589590726e-01 4.85139799052083930952505852474133e-02 2.73188521868607909690029877936013e-03
            2.36845394483151533426834589590726e-01 7.14640625611639990211187978275120e-01 2.73188521868607909690029877936013e-03
            4.85139799052083930952505852474133e-02 2.36845394483151533426834589590726e-01 2.73188521868607909690029877936013e-03
            4.85139799052083930952505852474133e-02 7.14640625611639990211187978275120e-01 2.73188521868607909690029877936013e-03
            7.14640625611639990211187978275120e-01 2.36845394483151533426834589590726e-01 2.73188521868607909690029877936013e-03
            7.14640625611639990211187978275120e-01 4.85139799052083930952505852474133e-02 2.73188521868607909690029877936013e-03
            4.14549152976443013862706266081659e-01 4.73723988093216474837277019105386e-03 1.00555100489236335162268609622060e-03
            4.14549152976443013862706266081659e-01 5.80713607142624876900072194985114e-01 1.00555100489236335162268609622060e-03
            4.73723988093216474837277019105386e-03 4.14549152976443013862706266081659e-01 1.00555100489236335162268609622060e-03
            4.73723988093216474837277019105386e-03 5.80713607142624876900072194985114e-01 1.00555100489236335162268609622060e-03
            5.80713607142624876900072194985114e-01 4.14549152976443013862706266081659e-01 1.00555100489236335162268609622060e-03
            5.80713607142624876900072194985114e-01 4.73723988093216474837277019105386e-03 1.00555100489236335162268609622060e-03
            3.07536188240941033011921490469831e-01 2.21352139073682022130107327484438e-02 2.13416735131724153926024989402777e-03
            3.07536188240941033011921490469831e-01 6.70328597851690743958386065060040e-01 2.13416735131724153926024989402777e-03
            2.21352139073682022130107327484438e-02 3.07536188240941033011921490469831e-01 2.13416735131724153926024989402777e-03
            2.21352139073682022130107327484438e-02 6.70328597851690743958386065060040e-01 2.13416735131724153926024989402777e-03
            6.70328597851690743958386065060040e-01 3.07536188240941033011921490469831e-01 2.13416735131724153926024989402777e-03
            6.70328597851690743958386065060040e-01 2.21352139073682022130107327484438e-02 2.13416735131724153926024989402777e-03
            1.10926664433470392734371046117303e-01 4.39339510946666188945242126351332e-03 6.43938204447683649848943598925644e-04
            1.10926664433470392734371046117303e-01 8.84679940457062885528216611419339e-01 6.43938204447683649848943598925644e-04
            4.39339510946666188945242126351332e-03 1.10926664433470392734371046117303e-01 6.43938204447683649848943598925644e-04
            4.39339510946666188945242126351332e-03 8.84679940457062885528216611419339e-01 6.43938204447683649848943598925644e-04
            8.84679940457062885528216611419339e-01 1.10926664433470392734371046117303e-01 6.43938204447683649848943598925644e-04
            8.84679940457062885528216611419339e-01 4.39339510946666188945242126351332e-03 6.43938204447683649848943598925644e-04
            3.18863122056367798151654824323487e-01 5.51602053562473687575717917752627e-02 3.29994166362838859882233855103095e-03
            3.18863122056367798151654824323487e-01 6.25976672587384874724136807344621e-01 3.29994166362838859882233855103095e-03
            5.51602053562473687575717917752627e-02 3.18863122056367798151654824323487e-01 3.29994166362838859882233855103095e-03
            5.51602053562473687575717917752627e-02 6.25976672587384874724136807344621e-01 3.29994166362838859882233855103095e-03
            6.25976672587384874724136807344621e-01 3.18863122056367798151654824323487e-01 3.29994166362838859882233855103095e-03
            6.25976672587384874724136807344621e-01 5.51602053562473687575717917752627e-02 3.29994166362838859882233855103095e-03
            3.95367759101382776343314162659226e-01 2.43998884696386975079462899884675e-02 2.47371640440409521627396216558736e-03
            3.95367759101382776343314162659226e-01 5.80232352428978526148739547352307e-01 2.47371640440409521627396216558736e-03
            2.43998884696386975079462899884675e-02 3.95367759101382776343314162659226e-01 2.47371640440409521627396216558736e-03
            2.43998884696386975079462899884675e-02 5.80232352428978526148739547352307e-01 2.47371640440409521627396216558736e-03
            5.80232352428978526148739547352307e-01 3.95367759101382776343314162659226e-01 2.47371640440409521627396216558736e-03
            5.80232352428978526148739547352307e-01 2.43998884696386975079462899884675e-02 2.47371640440409521627396216558736e-03
            2.13757830297353068615962001786102e-01 1.50634941829186158601672218537715e-01 4.59074077896355278727291704399249e-03
            2.13757830297353068615962001786102e-01 6.35607227873460800537941395305097e-01 4.59074077896355278727291704399249e-03
            1.50634941829186158601672218537715e-01 2.13757830297353068615962001786102e-01 4.59074077896355278727291704399249e-03
            1.50634941829186158601672218537715e-01 6.35607227873460800537941395305097e-01 4.59074077896355278727291704399249e-03
            6.35607227873460800537941395305097e-01 2.13757830297353068615962001786102e-01 4.59074077896355278727291704399249e-03
            6.35607227873460800537941395305097e-01 1.50634941829186158601672218537715e-01 4.59074077896355278727291704399249e-03
            2.47793702175406298227056112182254e-02 4.50238528829736278646933556046861e-03 3.22000887173716021996683345207657e-04
            2.47793702175406298227056112182254e-02 9.70718244494161996982484197360463e-01 3.22000887173716021996683345207657e-04
            4.50238528829736278646933556046861e-03 2.47793702175406298227056112182254e-02 3.22000887173716021996683345207657e-04
            4.50238528829736278646933556046861e-03 9.70718244494161996982484197360463e-01 3.22000887173716021996683345207657e-04
            9.70718244494161996982484197360463e-01 2.47793702175406298227056112182254e-02 3.22000887173716021996683345207657e-04
            9.70718244494161996982484197360463e-01 4.50238528829736278646933556046861e-03 3.22000887173716021996683345207657e-04
            ];
        
    case  34
        
        % ALG. DEG.:   34
        % PTS CARD.:  214
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.9e-15
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 6.05048246508555886274960400328382e-03
            4.59825951541271496503249238685385e-02 4.59825951541271496503249238685385e-02 1.14948328440455348198323548558619e-03
            4.59825951541271496503249238685385e-02 9.08034809691745659065986728819553e-01 1.14948328440455348198323548558619e-03
            9.08034809691745659065986728819553e-01 4.59825951541271496503249238685385e-02 1.14948328440455348198323548558619e-03
            2.08387475149684059383403678111790e-02 2.08387475149684059383403678111790e-02 5.49840283231598555419017415601957e-04
            2.08387475149684059383403678111790e-02 9.58322504970063215878894880006555e-01 5.49840283231598555419017415601957e-04
            9.58322504970063215878894880006555e-01 2.08387475149684059383403678111790e-02 5.49840283231598555419017415601957e-04
            7.57621741867907788625657872216834e-02 7.57621741867907788625657872216834e-02 1.99868672682467114765092475181518e-03
            7.57621741867907788625657872216834e-02 8.48475651626418470030444041185547e-01 1.99868672682467114765092475181518e-03
            8.48475651626418470030444041185547e-01 7.57621741867907788625657872216834e-02 1.99868672682467114765092475181518e-03
            3.76601980517545675830604068323737e-01 3.76601980517545675830604068323737e-01 5.87000329562878831596295370331973e-03
            3.76601980517545675830604068323737e-01 2.46796038964908648338791863352526e-01 5.87000329562878831596295370331973e-03
            2.46796038964908648338791863352526e-01 3.76601980517545675830604068323737e-01 5.87000329562878831596295370331973e-03
            2.87355839822164893515576977733872e-01 2.87355839822164893515576977733872e-01 5.88060381970681184810034736187845e-03
            2.87355839822164893515576977733872e-01 4.25288320355670212968846044532256e-01 5.88060381970681184810034736187845e-03
            4.25288320355670212968846044532256e-01 2.87355839822164893515576977733872e-01 5.88060381970681184810034736187845e-03
            4.74412498902845591874921638009255e-01 4.74412498902845591874921638009255e-01 3.24211260691925889226849832880362e-03
            4.74412498902845591874921638009255e-01 5.11750021943088162501567239814904e-02 3.24211260691925889226849832880362e-03
            5.11750021943088162501567239814904e-02 4.74412498902845591874921638009255e-01 3.24211260691925889226849832880362e-03
            1.11172701285845609842084513729787e-01 1.11172701285845609842084513729787e-01 2.91671683977711131835475200091423e-03
            1.11172701285845609842084513729787e-01 7.77654597428308780315830972540425e-01 2.91671683977711131835475200091423e-03
            7.77654597428308780315830972540425e-01 1.11172701285845609842084513729787e-01 2.91671683977711131835475200091423e-03
            4.91864961957689694749262798723066e-01 4.91864961957689694749262798723066e-01 1.86468547170908589480364803137036e-03
            4.91864961957689694749262798723066e-01 1.62700760846206105014744025538675e-02 1.86468547170908589480364803137036e-03
            1.62700760846206105014744025538675e-02 4.91864961957689694749262798723066e-01 1.86468547170908589480364803137036e-03
            1.50887722762872905635234133114864e-01 1.50887722762872905635234133114864e-01 3.84365534298183961908756955949684e-03
            1.50887722762872905635234133114864e-01 6.98224554474254244240682965028100e-01 3.84365534298183961908756955949684e-03
            6.98224554474254244240682965028100e-01 1.50887722762872905635234133114864e-01 3.84365534298183961908756955949684e-03
            4.15795739231892506199139702971479e-03 4.15795739231892506199139702971479e-03 1.13480678385471649041553454662079e-04
            4.15795739231892506199139702971479e-03 9.91684085215362109977377258474007e-01 1.13480678385471649041553454662079e-04
            9.91684085215362109977377258474007e-01 4.15795739231892506199139702971479e-03 1.13480678385471649041553454662079e-04
            2.40434910413816760321736865080311e-01 2.40434910413816760321736865080311e-01 5.40684965813481143576124665628413e-03
            2.40434910413816760321736865080311e-01 5.19130179172366479356526269839378e-01 5.40684965813481143576124665628413e-03
            5.19130179172366479356526269839378e-01 2.40434910413816760321736865080311e-01 5.40684965813481143576124665628413e-03
            4.15511596744837174988163042144151e-01 4.15511596744837174988163042144151e-01 5.32372242124446500705081675164365e-03
            4.15511596744837174988163042144151e-01 1.68976806510325650023673915711697e-01 5.32372242124446500705081675164365e-03
            1.68976806510325650023673915711697e-01 4.15511596744837174988163042144151e-01 5.32372242124446500705081675164365e-03
            4.99542182661511791064157250730204e-01 4.99542182661511791064157250730204e-01 3.80463083109050395971828528729475e-04
            4.99542182661511791064157250730204e-01 9.15634676976417871685498539591208e-04 3.80463083109050395971828528729475e-04
            9.15634676976417871685498539591208e-04 4.99542182661511791064157250730204e-01 3.80463083109050395971828528729475e-04
            4.48564265878708234858152081869775e-01 4.48564265878708234858152081869775e-01 4.43046965007651803725163830449674e-03
            4.48564265878708234858152081869775e-01 1.02871468242583530283695836260449e-01 4.43046965007651803725163830449674e-03
            1.02871468242583530283695836260449e-01 4.48564265878708234858152081869775e-01 4.43046965007651803725163830449674e-03
            1.94386543578966120016104923706735e-01 1.94386543578966120016104923706735e-01 4.70007051859917639180519444153106e-03
            1.94386543578966120016104923706735e-01 6.11226912842067759967790152586531e-01 4.70007051859917639180519444153106e-03
            6.11226912842067759967790152586531e-01 1.94386543578966120016104923706735e-01 4.70007051859917639180519444153106e-03
            3.85126251838280975192674304707907e-01 2.10967602701019364686008117359961e-02 5.08429861835656490701498833040262e-05
            3.85126251838280975192674304707907e-01 5.93776987891617036297020604251884e-01 5.08429861835656490701498833040262e-05
            2.10967602701019364686008117359961e-02 3.85126251838280975192674304707907e-01 5.08429861835656490701498833040262e-05
            2.10967602701019364686008117359961e-02 5.93776987891617036297020604251884e-01 5.08429861835656490701498833040262e-05
            5.93776987891617036297020604251884e-01 3.85126251838280975192674304707907e-01 5.08429861835656490701498833040262e-05
            5.93776987891617036297020604251884e-01 2.10967602701019364686008117359961e-02 5.08429861835656490701498833040262e-05
            5.28200673454987598698373574279685e-02 3.38811215039004848606496445029279e-03 3.27023983293753294551015065039223e-04
            5.28200673454987598698373574279685e-02 9.43791820504111256262547158257803e-01 3.27023983293753294551015065039223e-04
            3.38811215039004848606496445029279e-03 5.28200673454987598698373574279685e-02 3.27023983293753294551015065039223e-04
            3.38811215039004848606496445029279e-03 9.43791820504111256262547158257803e-01 3.27023983293753294551015065039223e-04
            9.43791820504111256262547158257803e-01 5.28200673454987598698373574279685e-02 3.27023983293753294551015065039223e-04
            9.43791820504111256262547158257803e-01 3.38811215039004848606496445029279e-03 3.27023983293753294551015065039223e-04
            5.08987652831316070289879860411020e-02 1.82083287994723903857607893996828e-02 7.59905829130818753942844256954459e-04
            5.08987652831316070289879860411020e-02 9.30892905917395929726865233533317e-01 7.59905829130818753942844256954459e-04
            1.82083287994723903857607893996828e-02 5.08987652831316070289879860411020e-02 7.59905829130818753942844256954459e-04
            1.82083287994723903857607893996828e-02 9.30892905917395929726865233533317e-01 7.59905829130818753942844256954459e-04
            9.30892905917395929726865233533317e-01 5.08987652831316070289879860411020e-02 7.59905829130818753942844256954459e-04
            9.30892905917395929726865233533317e-01 1.82083287994723903857607893996828e-02 7.59905829130818753942844256954459e-04
            1.29124372072608517569847208505962e-01 6.05107706433122560785164978369721e-02 2.33481884197296032587565228766380e-03
            1.29124372072608517569847208505962e-01 8.10364857284079254107211909285979e-01 2.33481884197296032587565228766380e-03
            6.05107706433122560785164978369721e-02 1.29124372072608517569847208505962e-01 2.33481884197296032587565228766380e-03
            6.05107706433122560785164978369721e-02 8.10364857284079254107211909285979e-01 2.33481884197296032587565228766380e-03
            8.10364857284079254107211909285979e-01 1.29124372072608517569847208505962e-01 2.33481884197296032587565228766380e-03
            8.10364857284079254107211909285979e-01 6.05107706433122560785164978369721e-02 2.33481884197296032587565228766380e-03
            8.78179027775244785969377403489489e-02 3.59198913635777883346378303031088e-02 1.47605781528108853661862553963147e-03
            8.78179027775244785969377403489489e-02 8.76262205858897802457363468420226e-01 1.47605781528108853661862553963147e-03
            3.59198913635777883346378303031088e-02 8.78179027775244785969377403489489e-02 1.47605781528108853661862553963147e-03
            3.59198913635777883346378303031088e-02 8.76262205858897802457363468420226e-01 1.47605781528108853661862553963147e-03
            8.76262205858897802457363468420226e-01 8.78179027775244785969377403489489e-02 1.47605781528108853661862553963147e-03
            8.76262205858897802457363468420226e-01 3.59198913635777883346378303031088e-02 1.47605781528108853661862553963147e-03
            2.73553738024819192986569760250859e-01 1.64061028670212660784955005510710e-01 4.93046515645235820801683956915440e-03
            2.73553738024819192986569760250859e-01 5.62385233304968146228475234238431e-01 4.93046515645235820801683956915440e-03
            1.64061028670212660784955005510710e-01 2.73553738024819192986569760250859e-01 4.93046515645235820801683956915440e-03
            1.64061028670212660784955005510710e-01 5.62385233304968146228475234238431e-01 4.93046515645235820801683956915440e-03
            5.62385233304968146228475234238431e-01 2.73553738024819192986569760250859e-01 4.93046515645235820801683956915440e-03
            5.62385233304968146228475234238431e-01 1.64061028670212660784955005510710e-01 4.93046515645235820801683956915440e-03
            9.44770656323741658821901978626556e-02 1.06832635235884227525859913043860e-02 8.63136629420848690806555847387926e-04
            9.44770656323741658821901978626556e-02 8.94839670844037482488886325882049e-01 8.63136629420848690806555847387926e-04
            1.06832635235884227525859913043860e-02 9.44770656323741658821901978626556e-02 8.63136629420848690806555847387926e-04
            1.06832635235884227525859913043860e-02 8.94839670844037482488886325882049e-01 8.63136629420848690806555847387926e-04
            8.94839670844037482488886325882049e-01 9.44770656323741658821901978626556e-02 8.63136629420848690806555847387926e-04
            8.94839670844037482488886325882049e-01 1.06832635235884227525859913043860e-02 8.63136629420848690806555847387926e-04
            4.11152233303600511682418527925620e-01 3.11326803063737105048414122165923e-02 2.51809125066696729705606472293766e-03
            4.11152233303600511682418527925620e-01 5.57715086390025760465505300089717e-01 2.51809125066696729705606472293766e-03
            3.11326803063737105048414122165923e-02 4.11152233303600511682418527925620e-01 2.51809125066696729705606472293766e-03
            3.11326803063737105048414122165923e-02 5.57715086390025760465505300089717e-01 2.51809125066696729705606472293766e-03
            5.57715086390025760465505300089717e-01 4.11152233303600511682418527925620e-01 2.51809125066696729705606472293766e-03
            5.57715086390025760465505300089717e-01 3.11326803063737105048414122165923e-02 2.51809125066696729705606472293766e-03
            3.59447569498255292685939821240027e-01 1.33192623221705830349748111984809e-01 4.82894011472339863139113802503743e-03
            3.59447569498255292685939821240027e-01 5.07359807280038932475463298032992e-01 4.82894011472339863139113802503743e-03
            1.33192623221705830349748111984809e-01 3.59447569498255292685939821240027e-01 4.82894011472339863139113802503743e-03
            1.33192623221705830349748111984809e-01 5.07359807280038932475463298032992e-01 4.82894011472339863139113802503743e-03
            5.07359807280038932475463298032992e-01 3.59447569498255292685939821240027e-01 4.82894011472339863139113802503743e-03
            5.07359807280038932475463298032992e-01 1.33192623221705830349748111984809e-01 4.82894011472339863139113802503743e-03
            3.88410085330437082618004751566332e-01 7.42973192372942126127099982113577e-02 3.80595751044293531886730796998108e-03
            3.88410085330437082618004751566332e-01 5.37292595432268704769285250222310e-01 3.80595751044293531886730796998108e-03
            7.42973192372942126127099982113577e-02 3.88410085330437082618004751566332e-01 3.80595751044293531886730796998108e-03
            7.42973192372942126127099982113577e-02 5.37292595432268704769285250222310e-01 3.80595751044293531886730796998108e-03
            5.37292595432268704769285250222310e-01 3.88410085330437082618004751566332e-01 3.80595751044293531886730796998108e-03
            5.37292595432268704769285250222310e-01 7.42973192372942126127099982113577e-02 3.80595751044293531886730796998108e-03
            3.25355043342057725919147515014629e-01 2.05166497814761955575946217322780e-01 5.53293194379955895201295135166220e-03
            3.25355043342057725919147515014629e-01 4.69478458843180290749330652033677e-01 5.53293194379955895201295135166220e-03
            2.05166497814761955575946217322780e-01 3.25355043342057725919147515014629e-01 5.53293194379955895201295135166220e-03
            2.05166497814761955575946217322780e-01 4.69478458843180290749330652033677e-01 5.53293194379955895201295135166220e-03
            4.69478458843180290749330652033677e-01 3.25355043342057725919147515014629e-01 5.53293194379955895201295135166220e-03
            4.69478458843180290749330652033677e-01 2.05166497814761955575946217322780e-01 5.53293194379955895201295135166220e-03
            1.74780479481068395930520864567370e-01 9.02196061251755165866228480808786e-02 3.25191985632487020468905924985847e-03
            1.74780479481068395930520864567370e-01 7.34999914393756115238431902980665e-01 3.25191985632487020468905924985847e-03
            9.02196061251755165866228480808786e-02 1.74780479481068395930520864567370e-01 3.25191985632487020468905924985847e-03
            9.02196061251755165866228480808786e-02 7.34999914393756115238431902980665e-01 3.25191985632487020468905924985847e-03
            7.34999914393756115238431902980665e-01 1.74780479481068395930520864567370e-01 3.25191985632487020468905924985847e-03
            7.34999914393756115238431902980665e-01 9.02196061251755165866228480808786e-02 3.25191985632487020468905924985847e-03
            1.94164618149563289462022908082872e-01 4.39158729045114326616783273493638e-02 2.41634362674254333028245333991890e-03
            1.94164618149563289462022908082872e-01 7.61919508945925194609571917681023e-01 2.41634362674254333028245333991890e-03
            4.39158729045114326616783273493638e-02 1.94164618149563289462022908082872e-01 2.41634362674254333028245333991890e-03
            4.39158729045114326616783273493638e-02 7.61919508945925194609571917681023e-01 2.41634362674254333028245333991890e-03
            7.61919508945925194609571917681023e-01 1.94164618149563289462022908082872e-01 2.41634362674254333028245333991890e-03
            7.61919508945925194609571917681023e-01 4.39158729045114326616783273493638e-02 2.41634362674254333028245333991890e-03
            2.48327873404578958194122151326155e-01 6.89638299646016944555171335196064e-02 3.29675723387471862266262512264348e-03
            2.48327873404578958194122151326155e-01 6.82708296630819333472572907339782e-01 3.29675723387471862266262512264348e-03
            6.89638299646016944555171335196064e-02 2.48327873404578958194122151326155e-01 3.29675723387471862266262512264348e-03
            6.89638299646016944555171335196064e-02 6.82708296630819333472572907339782e-01 3.29675723387471862266262512264348e-03
            6.82708296630819333472572907339782e-01 2.48327873404578958194122151326155e-01 3.29675723387471862266262512264348e-03
            6.82708296630819333472572907339782e-01 6.89638299646016944555171335196064e-02 3.29675723387471862266262512264348e-03
            2.22992041261200868440539579751203e-01 1.25181798142886657743133582698647e-01 4.14263325655032672162336737642363e-03
            2.22992041261200868440539579751203e-01 6.51826160595912473816326837550150e-01 4.14263325655032672162336737642363e-03
            1.25181798142886657743133582698647e-01 2.22992041261200868440539579751203e-01 4.14263325655032672162336737642363e-03
            1.25181798142886657743133582698647e-01 6.51826160595912473816326837550150e-01 4.14263325655032672162336737642363e-03
            6.51826160595912473816326837550150e-01 2.22992041261200868440539579751203e-01 4.14263325655032672162336737642363e-03
            6.51826160595912473816326837550150e-01 1.25181798142886657743133582698647e-01 4.14263325655032672162336737642363e-03
            1.49604423548951093181358373840339e-01 4.67029856128329395881237928733754e-03 6.99714692701543196015623315986431e-04
            1.49604423548951093181358373840339e-01 8.45725277889765592043147535150638e-01 6.99714692701543196015623315986431e-04
            4.67029856128329395881237928733754e-03 1.49604423548951093181358373840339e-01 6.99714692701543196015623315986431e-04
            4.67029856128329395881237928733754e-03 8.45725277889765592043147535150638e-01 6.99714692701543196015623315986431e-04
            8.45725277889765592043147535150638e-01 1.49604423548951093181358373840339e-01 6.99714692701543196015623315986431e-04
            8.45725277889765592043147535150638e-01 4.67029856128329395881237928733754e-03 6.99714692701543196015623315986431e-04
            3.28391413692126599155329813584103e-01 4.89259225038644121608122361521964e-02 3.04763201624416940963002708997465e-03
            3.28391413692126599155329813584103e-01 6.22682663804008940111600622913102e-01 3.04763201624416940963002708997465e-03
            4.89259225038644121608122361521964e-02 3.28391413692126599155329813584103e-01 3.04763201624416940963002708997465e-03
            4.89259225038644121608122361521964e-02 6.22682663804008940111600622913102e-01 3.04763201624416940963002708997465e-03
            6.22682663804008940111600622913102e-01 3.28391413692126599155329813584103e-01 3.04763201624416940963002708997465e-03
            6.22682663804008940111600622913102e-01 4.89259225038644121608122361521964e-02 3.04763201624416940963002708997465e-03
            1.42385500324542807470962202387454e-01 2.46582497782073459258178615982615e-02 1.57160307407964387055310240270956e-03
            1.42385500324542807470962202387454e-01 8.32956249897249922931052878993796e-01 1.57160307407964387055310240270956e-03
            2.46582497782073459258178615982615e-02 1.42385500324542807470962202387454e-01 1.57160307407964387055310240270956e-03
            2.46582497782073459258178615982615e-02 8.32956249897249922931052878993796e-01 1.57160307407964387055310240270956e-03
            8.32956249897249922931052878993796e-01 1.42385500324542807470962202387454e-01 1.57160307407964387055310240270956e-03
            8.32956249897249922931052878993796e-01 2.46582497782073459258178615982615e-02 1.57160307407964387055310240270956e-03
            4.23840299720368851588148118025856e-01 5.95994324974427527180642627513407e-03 1.11566589667664987396700215782630e-03
            4.23840299720368851588148118025856e-01 5.70199757029886900028259333339520e-01 1.11566589667664987396700215782630e-03
            5.95994324974427527180642627513407e-03 4.23840299720368851588148118025856e-01 1.11566589667664987396700215782630e-03
            5.95994324974427527180642627513407e-03 5.70199757029886900028259333339520e-01 1.11566589667664987396700215782630e-03
            5.70199757029886900028259333339520e-01 4.23840299720368851588148118025856e-01 1.11566589667664987396700215782630e-03
            5.70199757029886900028259333339520e-01 5.95994324974427527180642627513407e-03 1.11566589667664987396700215782630e-03
            2.67690419344305596816013803618262e-01 2.85187228422853940312808873613903e-02 2.21786548672097674414493972960827e-03
            2.67690419344305596816013803618262e-01 7.03790857813408998744364453159506e-01 2.21786548672097674414493972960827e-03
            2.85187228422853940312808873613903e-02 2.67690419344305596816013803618262e-01 2.21786548672097674414493972960827e-03
            2.85187228422853940312808873613903e-02 7.03790857813408998744364453159506e-01 2.21786548672097674414493972960827e-03
            7.03790857813408998744364453159506e-01 2.67690419344305596816013803618262e-01 2.21786548672097674414493972960827e-03
            7.03790857813408998744364453159506e-01 2.85187228422853940312808873613903e-02 2.21786548672097674414493972960827e-03
            3.45411378952901138017494986343081e-01 1.55626433010154947844672079781958e-02 1.74416367287191353269149018956341e-03
            3.45411378952901138017494986343081e-01 6.39025977746083273522970102931140e-01 1.74416367287191353269149018956341e-03
            1.55626433010154947844672079781958e-02 3.45411378952901138017494986343081e-01 1.74416367287191353269149018956341e-03
            1.55626433010154947844672079781958e-02 6.39025977746083273522970102931140e-01 1.74416367287191353269149018956341e-03
            6.39025977746083273522970102931140e-01 3.45411378952901138017494986343081e-01 1.74416367287191353269149018956341e-03
            6.39025977746083273522970102931140e-01 1.55626433010154947844672079781958e-02 1.74416367287191353269149018956341e-03
            3.53449519657663746574627339214203e-01 9.25718233933834830362563650396623e-04 3.65922936835055574028574554290572e-04
            3.53449519657663746574627339214203e-01 6.45624762108402472371437852416420e-01 3.65922936835055574028574554290572e-04
            9.25718233933834830362563650396623e-04 3.53449519657663746574627339214203e-01 3.65922936835055574028574554290572e-04
            9.25718233933834830362563650396623e-04 6.45624762108402472371437852416420e-01 3.65922936835055574028574554290572e-04
            6.45624762108402472371437852416420e-01 3.53449519657663746574627339214203e-01 3.65922936835055574028574554290572e-04
            6.45624762108402472371437852416420e-01 9.25718233933834830362563650396623e-04 3.65922936835055574028574554290572e-04
            3.03335639471789153542857775391894e-01 9.92862116185854320926651439549460e-02 4.12539871537647307581098132800435e-03
            3.03335639471789153542857775391894e-01 5.97378148909625372731113657209789e-01 4.12539871537647307581098132800435e-03
            9.92862116185854320926651439549460e-02 3.03335639471789153542857775391894e-01 4.12539871537647307581098132800435e-03
            9.92862116185854320926651439549460e-02 5.97378148909625372731113657209789e-01 4.12539871537647307581098132800435e-03
            5.97378148909625372731113657209789e-01 3.03335639471789153542857775391894e-01 4.12539871537647307581098132800435e-03
            5.97378148909625372731113657209789e-01 9.92862116185854320926651439549460e-02 4.12539871537647307581098132800435e-03
            2.17555603201839350968604946956475e-02 3.97450587251847731184994927389198e-03 2.48983786598558842154821357794958e-04
            2.17555603201839350968604946956475e-02 9.74269933807297650041334691195516e-01 2.48983786598558842154821357794958e-04
            3.97450587251847731184994927389198e-03 2.17555603201839350968604946956475e-02 2.48983786598558842154821357794958e-04
            3.97450587251847731184994927389198e-03 9.74269933807297650041334691195516e-01 2.48983786598558842154821357794958e-04
            9.74269933807297650041334691195516e-01 2.17555603201839350968604946956475e-02 2.48983786598558842154821357794958e-04
            9.74269933807297650041334691195516e-01 3.97450587251847731184994927389198e-03 2.48983786598558842154821357794958e-04
            2.07546868273520451753100246605754e-01 1.37796394053688077419028701342540e-02 1.40290906865686267404602638464439e-03
            2.07546868273520451753100246605754e-01 7.78673492321110716218868219584692e-01 1.40290906865686267404602638464439e-03
            1.37796394053688077419028701342540e-02 2.07546868273520451753100246605754e-01 1.40290906865686267404602638464439e-03
            1.37796394053688077419028701342540e-02 7.78673492321110716218868219584692e-01 1.40290906865686267404602638464439e-03
            7.78673492321110716218868219584692e-01 2.07546868273520451753100246605754e-01 1.40290906865686267404602638464439e-03
            7.78673492321110716218868219584692e-01 1.37796394053688077419028701342540e-02 1.40290906865686267404602638464439e-03
            2.13560239517947836329625488360762e-01 7.72139120327043496430641233985170e-04 2.88825287117379120738897002595991e-04
            2.13560239517947836329625488360762e-01 7.85667621361725077022697405482177e-01 2.88825287117379120738897002595991e-04
            7.72139120327043496430641233985170e-04 2.13560239517947836329625488360762e-01 2.88825287117379120738897002595991e-04
            7.72139120327043496430641233985170e-04 7.85667621361725077022697405482177e-01 2.88825287117379120738897002595991e-04
            7.85667621361725077022697405482177e-01 2.13560239517947836329625488360762e-01 2.88825287117379120738897002595991e-04
            7.85667621361725077022697405482177e-01 7.72139120327043496430641233985170e-04 2.88825287117379120738897002595991e-04
            2.79386431782223720876601191775990e-01 5.46179089429344854361714212132028e-03 9.81314402741667981058815151129693e-04
            2.79386431782223720876601191775990e-01 7.15151777323482895631912015232956e-01 9.81314402741667981058815151129693e-04
            5.46179089429344854361714212132028e-03 2.79386431782223720876601191775990e-01 9.81314402741667981058815151129693e-04
            5.46179089429344854361714212132028e-03 7.15151777323482895631912015232956e-01 9.81314402741667981058815151129693e-04
            7.15151777323482895631912015232956e-01 2.79386431782223720876601191775990e-01 9.81314402741667981058815151129693e-04
            7.15151777323482895631912015232956e-01 5.46179089429344854361714212132028e-03 9.81314402741667981058815151129693e-04
            9.68669225755471735572754710119625e-02 4.70969101854539203629197718914412e-06 1.43672677304066987455724579980654e-04
            9.68669225755471735572754710119625e-02 9.03128367733434300745898326567840e-01 1.43672677304066987455724579980654e-04
            4.70969101854539203629197718914412e-06 9.68669225755471735572754710119625e-02 1.43672677304066987455724579980654e-04
            4.70969101854539203629197718914412e-06 9.03128367733434300745898326567840e-01 1.43672677304066987455724579980654e-04
            9.03128367733434300745898326567840e-01 9.68669225755471735572754710119625e-02 1.43672677304066987455724579980654e-04
            9.03128367733434300745898326567840e-01 4.70969101854539203629197718914412e-06 1.43672677304066987455724579980654e-04
            ];
        
    case  35
        
        % ALG. DEG.:   35
        % PTS CARD.:  228
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.7e-15
        
        xyw=[
            3.01435168617317950090495060067042e-01 3.01435168617317950090495060067042e-01 2.87396701483509002875149640487962e-03
            3.01435168617317950090495060067042e-01 3.97129662765364099819009879865916e-01 2.87396701483509002875149640487962e-03
            3.97129662765364099819009879865916e-01 3.01435168617317950090495060067042e-01 2.87396701483509002875149640487962e-03
            4.91641403982890690116391851915978e-01 4.91641403982890690116391851915978e-01 1.14818404604971461592521908556819e-03
            4.91641403982890690116391851915978e-01 1.67171920342186197672162961680442e-02 1.14818404604971461592521908556819e-03
            1.67171920342186197672162961680442e-02 4.91641403982890690116391851915978e-01 1.14818404604971461592521908556819e-03
            3.57793135798597061736359137285035e-01 3.57793135798597061736359137285035e-01 3.99767542093215459686472712519389e-03
            3.57793135798597061736359137285035e-01 2.84413728402805876527281725429930e-01 3.99767542093215459686472712519389e-03
            2.84413728402805876527281725429930e-01 3.57793135798597061736359137285035e-01 3.99767542093215459686472712519389e-03
            2.62413960109345423266091756886453e-01 2.62413960109345423266091756886453e-01 4.39509294666389632721958946603991e-03
            2.62413960109345423266091756886453e-01 4.75172079781309153467816486227093e-01 4.39509294666389632721958946603991e-03
            4.75172079781309153467816486227093e-01 2.62413960109345423266091756886453e-01 4.39509294666389632721958946603991e-03
            4.02816450585781860027623224596027e-01 4.02816450585781860027623224596027e-01 4.16018330616281083872420865077402e-03
            4.02816450585781860027623224596027e-01 1.94367098828436279944753550807945e-01 4.16018330616281083872420865077402e-03
            1.94367098828436279944753550807945e-01 4.02816450585781860027623224596027e-01 4.16018330616281083872420865077402e-03
            4.73088965500664149210763298469828e-01 4.73088965500664149210763298469828e-01 2.63759414258074543699339464808418e-03
            4.73088965500664149210763298469828e-01 5.38220689986717015784734030603431e-02 2.63759414258074543699339464808418e-03
            5.38220689986717015784734030603431e-02 4.73088965500664149210763298469828e-01 2.63759414258074543699339464808418e-03
            6.28680490440657047113148792050197e-02 6.28680490440657047113148792050197e-02 1.48410331952580042125244830231168e-03
            6.28680490440657047113148792050197e-02 8.74263901911868535066219010332134e-01 1.48410331952580042125244830231168e-03
            8.74263901911868535066219010332134e-01 6.28680490440657047113148792050197e-02 1.48410331952580042125244830231168e-03
            4.41597433754930324134591046458809e-01 4.41597433754930324134591046458809e-01 3.80259126526817647917511067134910e-03
            4.41597433754930324134591046458809e-01 1.16805132490139351730817907082383e-01 3.80259126526817647917511067134910e-03
            1.16805132490139351730817907082383e-01 4.41597433754930324134591046458809e-01 3.80259126526817647917511067134910e-03
            1.96271599834707231868602406166247e-02 1.96271599834707231868602406166247e-02 5.06560516369039190442968845218275e-04
            1.96271599834707231868602406166247e-02 9.60745680033058602198536846117349e-01 5.06560516369039190442968845218275e-04
            9.60745680033058602198536846117349e-01 1.96271599834707231868602406166247e-02 5.06560516369039190442968845218275e-04
            4.02868688301933966511292339873762e-02 4.02868688301933966511292339873762e-02 1.03241781329916270867896876950454e-03
            4.02868688301933966511292339873762e-02 9.19426262339613220575529339839704e-01 1.03241781329916270867896876950454e-03
            9.19426262339613220575529339839704e-01 4.02868688301933966511292339873762e-02 1.03241781329916270867896876950454e-03
            9.85261546687923595033709034396452e-02 9.85261546687923595033709034396452e-02 2.50181980777866531592823484686505e-03
            9.85261546687923595033709034396452e-02 8.02947690662415336504409424378537e-01 2.50181980777866531592823484686505e-03
            8.02947690662415336504409424378537e-01 9.85261546687923595033709034396452e-02 2.50181980777866531592823484686505e-03
            1.39534462989307272984262908721576e-01 1.39534462989307272984262908721576e-01 3.35953990620869481775745413187906e-03
            1.39534462989307272984262908721576e-01 7.20931074021385454031474182556849e-01 3.35953990620869481775745413187906e-03
            7.20931074021385454031474182556849e-01 1.39534462989307272984262908721576e-01 3.35953990620869481775745413187906e-03
            2.21536010509691960734812710143160e-01 2.21536010509691960734812710143160e-01 4.58095447046864921031295025954932e-03
            2.21536010509691960734812710143160e-01 5.56927978980616078530374579713680e-01 4.58095447046864921031295025954932e-03
            5.56927978980616078530374579713680e-01 2.21536010509691960734812710143160e-01 4.58095447046864921031295025954932e-03
            1.80750322135299568149235938108177e-01 1.80750322135299568149235938108177e-01 4.18595331850647994070824253753926e-03
            1.80750322135299568149235938108177e-01 6.38499355729400863701528123783646e-01 4.18595331850647994070824253753926e-03
            6.38499355729400863701528123783646e-01 1.80750322135299568149235938108177e-01 4.18595331850647994070824253753926e-03
            4.27720726848295623723217318001844e-03 4.27720726848295623723217318001844e-03 1.20166868702420545807207918898030e-04
            4.27720726848295623723217318001844e-03 9.91445585463034073647747845825506e-01 1.20166868702420545807207918898030e-04
            9.91445585463034073647747845825506e-01 4.27720726848295623723217318001844e-03 1.20166868702420545807207918898030e-04
            4.99913222486225106155899311488611e-01 4.99913222486225106155899311488611e-01 3.13792170535282313736935577708209e-04
            4.99913222486225106155899311488611e-01 1.73555027549787688201377022778615e-04 3.13792170535282313736935577708209e-04
            1.73555027549787688201377022778615e-04 4.99913222486225106155899311488611e-01 3.13792170535282313736935577708209e-04
            5.50454796915316388328243135674711e-02 2.06822021540594260316514585440473e-03 2.27099691187698603635561345193139e-04
            5.50454796915316388328243135674711e-02 9.42886300093062401650456649804255e-01 2.27099691187698603635561345193139e-04
            2.06822021540594260316514585440473e-03 5.50454796915316388328243135674711e-02 2.27099691187698603635561345193139e-04
            2.06822021540594260316514585440473e-03 9.42886300093062401650456649804255e-01 2.27099691187698603635561345193139e-04
            9.42886300093062401650456649804255e-01 5.50454796915316388328243135674711e-02 2.27099691187698603635561345193139e-04
            9.42886300093062401650456649804255e-01 2.06822021540594260316514585440473e-03 2.27099691187698603635561345193139e-04
            4.45376464915297476387934239028255e-01 8.90499418612781569459802000210402e-03 9.85874665492084698362185157805015e-04
            4.45376464915297476387934239028255e-01 5.45718540898574744346660736482590e-01 9.85874665492084698362185157805015e-04
            8.90499418612781569459802000210402e-03 4.45376464915297476387934239028255e-01 9.85874665492084698362185157805015e-04
            8.90499418612781569459802000210402e-03 5.45718540898574744346660736482590e-01 9.85874665492084698362185157805015e-04
            5.45718540898574744346660736482590e-01 4.45376464915297476387934239028255e-01 9.85874665492084698362185157805015e-04
            5.45718540898574744346660736482590e-01 8.90499418612781569459802000210402e-03 9.85874665492084698362185157805015e-04
            1.52384002524315248505715203464206e-01 2.11597756303986755596624469433209e-03 3.91333152719676587941860512742664e-04
            1.52384002524315248505715203464206e-01 8.45500019912644895647702014684910e-01 3.91333152719676587941860512742664e-04
            2.11597756303986755596624469433209e-03 1.52384002524315248505715203464206e-01 3.91333152719676587941860512742664e-04
            2.11597756303986755596624469433209e-03 8.45500019912644895647702014684910e-01 3.91333152719676587941860512742664e-04
            8.45500019912644895647702014684910e-01 1.52384002524315248505715203464206e-01 3.91333152719676587941860512742664e-04
            8.45500019912644895647702014684910e-01 2.11597756303986755596624469433209e-03 3.91333152719676587941860512742664e-04
            3.97516610028231298645806646163692e-01 3.26127342913124992906936938652507e-03 6.32346551198585031874910278304469e-04
            3.97516610028231298645806646163692e-01 5.99222116542637461833464840310626e-01 6.32346551198585031874910278304469e-04
            3.26127342913124992906936938652507e-03 3.97516610028231298645806646163692e-01 6.32346551198585031874910278304469e-04
            3.26127342913124992906936938652507e-03 5.99222116542637461833464840310626e-01 6.32346551198585031874910278304469e-04
            5.99222116542637461833464840310626e-01 3.97516610028231298645806646163692e-01 6.32346551198585031874910278304469e-04
            5.99222116542637461833464840310626e-01 3.26127342913124992906936938652507e-03 6.32346551198585031874910278304469e-04
            1.56078769403476091959248606144683e-01 4.51678948131286547851281909515819e-02 1.77862813045475548490159667380794e-03
            1.56078769403476091959248606144683e-01 7.98753335783395246316729298996506e-01 1.77862813045475548490159667380794e-03
            4.51678948131286547851281909515819e-02 1.56078769403476091959248606144683e-01 1.77862813045475548490159667380794e-03
            4.51678948131286547851281909515819e-02 7.98753335783395246316729298996506e-01 1.77862813045475548490159667380794e-03
            7.98753335783395246316729298996506e-01 1.56078769403476091959248606144683e-01 1.77862813045475548490159667380794e-03
            7.98753335783395246316729298996506e-01 4.51678948131286547851281909515819e-02 1.77862813045475548490159667380794e-03
            1.97295676276558723527543293130293e-01 8.60602170575682987041510330072924e-03 9.21416331587752469867158566785292e-04
            1.97295676276558723527543293130293e-01 7.94098302017684387621443420357537e-01 9.21416331587752469867158566785292e-04
            8.60602170575682987041510330072924e-03 1.97295676276558723527543293130293e-01 9.21416331587752469867158566785292e-04
            8.60602170575682987041510330072924e-03 7.94098302017684387621443420357537e-01 9.21416331587752469867158566785292e-04
            7.94098302017684387621443420357537e-01 1.97295676276558723527543293130293e-01 9.21416331587752469867158566785292e-04
            7.94098302017684387621443420357537e-01 8.60602170575682987041510330072924e-03 9.21416331587752469867158566785292e-04
            5.00938368999261987579707522399985e-02 1.39743532515683617500901192443052e-02 6.64489967827079099603471146195943e-04
            5.00938368999261987579707522399985e-02 9.35931809848505436022492176562082e-01 6.64489967827079099603471146195943e-04
            1.39743532515683617500901192443052e-02 5.00938368999261987579707522399985e-02 6.64489967827079099603471146195943e-04
            1.39743532515683617500901192443052e-02 9.35931809848505436022492176562082e-01 6.64489967827079099603471146195943e-04
            9.35931809848505436022492176562082e-01 5.00938368999261987579707522399985e-02 6.64489967827079099603471146195943e-04
            9.35931809848505436022492176562082e-01 1.39743532515683617500901192443052e-02 6.64489967827079099603471146195943e-04
            2.98859063014319081830194591020700e-01 1.87471332499894505208004602536676e-01 4.51170642755310997984441456765126e-03
            2.98859063014319081830194591020700e-01 5.13669604485786468472952037700452e-01 4.51170642755310997984441456765126e-03
            1.87471332499894505208004602536676e-01 2.98859063014319081830194591020700e-01 4.51170642755310997984441456765126e-03
            1.87471332499894505208004602536676e-01 5.13669604485786468472952037700452e-01 4.51170642755310997984441456765126e-03
            5.13669604485786468472952037700452e-01 2.98859063014319081830194591020700e-01 4.51170642755310997984441456765126e-03
            5.13669604485786468472952037700452e-01 1.87471332499894505208004602536676e-01 4.51170642755310997984441456765126e-03
            3.34763343265331358988845522617339e-01 2.32722344075948522901242654370435e-01 4.30275224704192334190810953487016e-03
            3.34763343265331358988845522617339e-01 4.32514312658720090354336207383312e-01 4.30275224704192334190810953487016e-03
            2.32722344075948522901242654370435e-01 3.34763343265331358988845522617339e-01 4.30275224704192334190810953487016e-03
            2.32722344075948522901242654370435e-01 4.32514312658720090354336207383312e-01 4.30275224704192334190810953487016e-03
            4.32514312658720090354336207383312e-01 3.34763343265331358988845522617339e-01 4.30275224704192334190810953487016e-03
            4.32514312658720090354336207383312e-01 2.32722344075948522901242654370435e-01 4.30275224704192334190810953487016e-03
            2.09372824976375354610880208383605e-01 2.90018846146845241107392610047100e-02 1.80523399849691606328105919487825e-03
            2.09372824976375354610880208383605e-01 7.61625290408940114339486626704456e-01 1.80523399849691606328105919487825e-03
            2.90018846146845241107392610047100e-02 2.09372824976375354610880208383605e-01 1.80523399849691606328105919487825e-03
            2.90018846146845241107392610047100e-02 7.61625290408940114339486626704456e-01 1.80523399849691606328105919487825e-03
            7.61625290408940114339486626704456e-01 2.09372824976375354610880208383605e-01 1.80523399849691606328105919487825e-03
            7.61625290408940114339486626704456e-01 2.90018846146845241107392610047100e-02 1.80523399849691606328105919487825e-03
            1.06564744861383425833523119763413e-01 5.35674828677535283527610943110631e-02 1.76178038360539799753179401875514e-03
            1.06564744861383425833523119763413e-01 8.39867772270863066630397497647209e-01 1.76178038360539799753179401875514e-03
            5.35674828677535283527610943110631e-02 1.06564744861383425833523119763413e-01 1.76178038360539799753179401875514e-03
            5.35674828677535283527610943110631e-02 8.39867772270863066630397497647209e-01 1.76178038360539799753179401875514e-03
            8.39867772270863066630397497647209e-01 1.06564744861383425833523119763413e-01 1.76178038360539799753179401875514e-03
            8.39867772270863066630397497647209e-01 5.35674828677535283527610943110631e-02 1.76178038360539799753179401875514e-03
            3.21362627669543321484724174297298e-01 3.49619434914352376561619273331871e-03 6.89038261003182715344617115249548e-04
            3.21362627669543321484724174297298e-01 6.75141177981313123090956196392654e-01 6.89038261003182715344617115249548e-04
            3.49619434914352376561619273331871e-03 3.21362627669543321484724174297298e-01 6.89038261003182715344617115249548e-04
            3.49619434914352376561619273331871e-03 6.75141177981313123090956196392654e-01 6.89038261003182715344617115249548e-04
            6.75141177981313123090956196392654e-01 3.21362627669543321484724174297298e-01 6.89038261003182715344617115249548e-04
            6.75141177981313123090956196392654e-01 3.49619434914352376561619273331871e-03 6.89038261003182715344617115249548e-04
            2.41587026347987537944206337670039e-01 1.17604279684175661371825505341349e-03 3.47514792975928241310079602754968e-04
            2.41587026347987537944206337670039e-01 7.57236930855170631282646809268044e-01 3.47514792975928241310079602754968e-04
            1.17604279684175661371825505341349e-03 2.41587026347987537944206337670039e-01 3.47514792975928241310079602754968e-04
            1.17604279684175661371825505341349e-03 7.57236930855170631282646809268044e-01 3.47514792975928241310079602754968e-04
            7.57236930855170631282646809268044e-01 2.41587026347987537944206337670039e-01 3.47514792975928241310079602754968e-04
            7.57236930855170631282646809268044e-01 1.17604279684175661371825505341349e-03 3.47514792975928241310079602754968e-04
            3.33153099720764844793308157022693e-01 1.13953928744148377805167626775074e-01 4.06486029903899495629993054990337e-03
            3.33153099720764844793308157022693e-01 5.52892971535086874546038870903431e-01 4.06486029903899495629993054990337e-03
            1.13953928744148377805167626775074e-01 3.33153099720764844793308157022693e-01 4.06486029903899495629993054990337e-03
            1.13953928744148377805167626775074e-01 5.52892971535086874546038870903431e-01 4.06486029903899495629993054990337e-03
            5.52892971535086874546038870903431e-01 3.33153099720764844793308157022693e-01 4.06486029903899495629993054990337e-03
            5.52892971535086874546038870903431e-01 1.13953928744148377805167626775074e-01 4.06486029903899495629993054990337e-03
            1.55116554813303442195149273175048e-01 8.49385995115550379574997919007728e-02 2.74482817993335331124837850325093e-03
            1.55116554813303442195149273175048e-01 7.59944845675141533725138742738636e-01 2.74482817993335331124837850325093e-03
            8.49385995115550379574997919007728e-02 1.55116554813303442195149273175048e-01 2.74482817993335331124837850325093e-03
            8.49385995115550379574997919007728e-02 7.59944845675141533725138742738636e-01 2.74482817993335331124837850325093e-03
            7.59944845675141533725138742738636e-01 1.55116554813303442195149273175048e-01 2.74482817993335331124837850325093e-03
            7.59944845675141533725138742738636e-01 8.49385995115550379574997919007728e-02 2.74482817993335331124837850325093e-03
            3.71014151677461878620079005486332e-01 1.52565186347216358297984584169171e-01 4.11799366142912449256785478723941e-03
            3.71014151677461878620079005486332e-01 4.76420661975321735326360794715583e-01 4.11799366142912449256785478723941e-03
            1.52565186347216358297984584169171e-01 3.71014151677461878620079005486332e-01 4.11799366142912449256785478723941e-03
            1.52565186347216358297984584169171e-01 4.76420661975321735326360794715583e-01 4.11799366142912449256785478723941e-03
            4.76420661975321735326360794715583e-01 3.71014151677461878620079005486332e-01 4.11799366142912449256785478723941e-03
            4.76420661975321735326360794715583e-01 1.52565186347216358297984584169171e-01 4.11799366142912449256785478723941e-03
            2.70883661807772069263222647350631e-01 1.36186781090766283186521690140580e-02 1.38988675968128349083596795310314e-03
            2.70883661807772069263222647350631e-01 7.15497660083151298948678231681697e-01 1.38988675968128349083596795310314e-03
            1.36186781090766283186521690140580e-02 2.70883661807772069263222647350631e-01 1.38988675968128349083596795310314e-03
            1.36186781090766283186521690140580e-02 7.15497660083151298948678231681697e-01 1.38988675968128349083596795310314e-03
            7.15497660083151298948678231681697e-01 2.70883661807772069263222647350631e-01 1.38988675968128349083596795310314e-03
            7.15497660083151298948678231681697e-01 1.36186781090766283186521690140580e-02 1.38988675968128349083596795310314e-03
            3.52506748415466475421453651506454e-01 5.62263874435523919315116359030071e-02 3.16589094074169784612249323174638e-03
            3.52506748415466475421453651506454e-01 5.91266864140981174280398136033909e-01 3.16589094074169784612249323174638e-03
            5.62263874435523919315116359030071e-02 3.52506748415466475421453651506454e-01 3.16589094074169784612249323174638e-03
            5.62263874435523919315116359030071e-02 5.91266864140981174280398136033909e-01 3.16589094074169784612249323174638e-03
            5.91266864140981174280398136033909e-01 3.52506748415466475421453651506454e-01 3.16589094074169784612249323174638e-03
            5.91266864140981174280398136033909e-01 5.62263874435523919315116359030071e-02 3.16589094074169784612249323174638e-03
            4.05720132778190467615075931462343e-01 8.17534894396882150235938979676575e-02 3.47285535997079227582551297359714e-03
            4.05720132778190467615075931462343e-01 5.12526377782121289605754554941086e-01 3.47285535997079227582551297359714e-03
            8.17534894396882150235938979676575e-02 4.05720132778190467615075931462343e-01 3.47285535997079227582551297359714e-03
            8.17534894396882150235938979676575e-02 5.12526377782121289605754554941086e-01 3.47285535997079227582551297359714e-03
            5.12526377782121289605754554941086e-01 4.05720132778190467615075931462343e-01 3.47285535997079227582551297359714e-03
            5.12526377782121289605754554941086e-01 8.17534894396882150235938979676575e-02 3.47285535997079227582551297359714e-03
            2.18949681014998343009381187584950e-01 6.36424301897473843059316322978702e-02 2.78297055784304482248248113762656e-03
            2.18949681014998343009381187584950e-01 7.17407888795254300440262795746094e-01 2.78297055784304482248248113762656e-03
            6.36424301897473843059316322978702e-02 2.18949681014998343009381187584950e-01 2.78297055784304482248248113762656e-03
            6.36424301897473843059316322978702e-02 7.17407888795254300440262795746094e-01 2.78297055784304482248248113762656e-03
            7.17407888795254300440262795746094e-01 2.18949681014998343009381187584950e-01 2.78297055784304482248248113762656e-03
            7.17407888795254300440262795746094e-01 6.36424301897473843059316322978702e-02 2.78297055784304482248248113762656e-03
            2.06709248085564512020795291391551e-01 1.15394220836074104519219929443352e-01 3.66880507838133768788724253795408e-03
            2.06709248085564512020795291391551e-01 6.77896531078361341826621355721727e-01 3.66880507838133768788724253795408e-03
            1.15394220836074104519219929443352e-01 2.06709248085564512020795291391551e-01 3.66880507838133768788724253795408e-03
            1.15394220836074104519219929443352e-01 6.77896531078361341826621355721727e-01 3.66880507838133768788724253795408e-03
            6.77896531078361341826621355721727e-01 2.06709248085564512020795291391551e-01 3.66880507838133768788724253795408e-03
            6.77896531078361341826621355721727e-01 1.15394220836074104519219929443352e-01 3.66880507838133768788724253795408e-03
            2.56601087879092248922319186021923e-01 1.48187116283310715836307736026356e-01 4.35571331178043284065859452880431e-03
            2.56601087879092248922319186021923e-01 5.95211795837596979730221846693894e-01 4.35571331178043284065859452880431e-03
            1.48187116283310715836307736026356e-01 2.56601087879092248922319186021923e-01 4.35571331178043284065859452880431e-03
            1.48187116283310715836307736026356e-01 5.95211795837596979730221846693894e-01 4.35571331178043284065859452880431e-03
            5.95211795837596979730221846693894e-01 2.56601087879092248922319186021923e-01 4.35571331178043284065859452880431e-03
            5.95211795837596979730221846693894e-01 1.48187116283310715836307736026356e-01 4.35571331178043284065859452880431e-03
            1.38206295211249968746258787177794e-01 1.85978710613813984220410446823735e-02 1.36837471636100881620945468597483e-03
            1.38206295211249968746258787177794e-01 8.43195833727368615484465408371761e-01 1.36837471636100881620945468597483e-03
            1.85978710613813984220410446823735e-02 1.38206295211249968746258787177794e-01 1.36837471636100881620945468597483e-03
            1.85978710613813984220410446823735e-02 8.43195833727368615484465408371761e-01 1.36837471636100881620945468597483e-03
            8.43195833727368615484465408371761e-01 1.38206295211249968746258787177794e-01 1.36837471636100881620945468597483e-03
            8.43195833727368615484465408371761e-01 1.85978710613813984220410446823735e-02 1.36837471636100881620945468597483e-03
            4.25810208585481020371332760987571e-01 3.29710199269463780091804494531971e-02 2.35082879046609070569640032033476e-03
            4.25810208585481020371332760987571e-01 5.41218771487572580802805077837547e-01 2.35082879046609070569640032033476e-03
            3.29710199269463780091804494531971e-02 4.25810208585481020371332760987571e-01 2.35082879046609070569640032033476e-03
            3.29710199269463780091804494531971e-02 5.41218771487572580802805077837547e-01 2.35082879046609070569640032033476e-03
            5.41218771487572580802805077837547e-01 4.25810208585481020371332760987571e-01 2.35082879046609070569640032033476e-03
            5.41218771487572580802805077837547e-01 3.29710199269463780091804494531971e-02 2.35082879046609070569640032033476e-03
            2.79725926952240189216070120892255e-01 8.53795515729538628812989031757752e-02 3.58750447996635662972764180267404e-03
            2.79725926952240189216070120892255e-01 6.34894521474805961780418783746427e-01 3.58750447996635662972764180267404e-03
            8.53795515729538628812989031757752e-02 2.79725926952240189216070120892255e-01 3.58750447996635662972764180267404e-03
            8.53795515729538628812989031757752e-02 6.34894521474805961780418783746427e-01 3.58750447996635662972764180267404e-03
            6.34894521474805961780418783746427e-01 2.79725926952240189216070120892255e-01 3.58750447996635662972764180267404e-03
            6.34894521474805961780418783746427e-01 8.53795515729538628812989031757752e-02 3.58750447996635662972764180267404e-03
            2.24666835802111858333418581423757e-02 3.63305584293396573516976566509129e-03 2.36623486900244448492836024655617e-04
            2.24666835802111858333418581423757e-02 9.73900260576854859273510101047577e-01 2.36623486900244448492836024655617e-04
            3.63305584293396573516976566509129e-03 2.24666835802111858333418581423757e-02 2.36623486900244448492836024655617e-04
            3.63305584293396573516976566509129e-03 9.73900260576854859273510101047577e-01 2.36623486900244448492836024655617e-04
            9.73900260576854859273510101047577e-01 2.24666835802111858333418581423757e-02 2.36623486900244448492836024655617e-04
            9.73900260576854859273510101047577e-01 3.63305584293396573516976566509129e-03 2.36623486900244448492836024655617e-04
            8.38622585244554186179044563687057e-02 2.60200776623639613849281460034035e-02 1.26733105568389002072715587843277e-03
            8.38622585244554186179044563687057e-02 8.90117663813180692855553388653789e-01 1.26733105568389002072715587843277e-03
            2.60200776623639613849281460034035e-02 8.38622585244554186179044563687057e-02 1.26733105568389002072715587843277e-03
            2.60200776623639613849281460034035e-02 8.90117663813180692855553388653789e-01 1.26733105568389002072715587843277e-03
            8.90117663813180692855553388653789e-01 8.38622585244554186179044563687057e-02 1.26733105568389002072715587843277e-03
            8.90117663813180692855553388653789e-01 2.60200776623639613849281460034035e-02 1.26733105568389002072715587843277e-03
            2.84800346414468275035858368937625e-01 3.94569261564381226770059640784893e-02 2.53507865000569820868037140826345e-03
            2.84800346414468275035858368937625e-01 6.75742727429093581470453955262201e-01 2.53507865000569820868037140826345e-03
            3.94569261564381226770059640784893e-02 2.84800346414468275035858368937625e-01 2.53507865000569820868037140826345e-03
            3.94569261564381226770059640784893e-02 6.75742727429093581470453955262201e-01 2.53507865000569820868037140826345e-03
            6.75742727429093581470453955262201e-01 2.84800346414468275035858368937625e-01 2.53507865000569820868037140826345e-03
            6.75742727429093581470453955262201e-01 3.94569261564381226770059640784893e-02 2.53507865000569820868037140826345e-03
            3.56594103536573836521483826800250e-01 1.97707177044081881811532497295048e-02 2.02703604344201156195470758802912e-03
            3.56594103536573836521483826800250e-01 6.23635178759017971827915971516632e-01 2.02703604344201156195470758802912e-03
            1.97707177044081881811532497295048e-02 3.56594103536573836521483826800250e-01 2.02703604344201156195470758802912e-03
            1.97707177044081881811532497295048e-02 6.23635178759017971827915971516632e-01 2.02703604344201156195470758802912e-03
            6.23635178759017971827915971516632e-01 3.56594103536573836521483826800250e-01 2.02703604344201156195470758802912e-03
            6.23635178759017971827915971516632e-01 1.97707177044081881811532497295048e-02 2.02703604344201156195470758802912e-03
            9.74242311151380380174202855414478e-02 5.02869741562738704898638530949029e-03 6.27239193620508111472189316515369e-04
            9.74242311151380380174202855414478e-02 8.97547071469234514218271669960814e-01 6.27239193620508111472189316515369e-04
            5.02869741562738704898638530949029e-03 9.74242311151380380174202855414478e-02 6.27239193620508111472189316515369e-04
            5.02869741562738704898638530949029e-03 8.97547071469234514218271669960814e-01 6.27239193620508111472189316515369e-04
            8.97547071469234514218271669960814e-01 9.74242311151380380174202855414478e-02 6.27239193620508111472189316515369e-04
            8.97547071469234514218271669960814e-01 5.02869741562738704898638530949029e-03 6.27239193620508111472189316515369e-04
            ];
        
    case  36
        
        % ALG. DEG.:   36
        % PTS CARD.:  243
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.2e-15
        
        xyw=[
            4.12357340313630305050907054464915e-01 4.12357340313630305050907054464915e-01 3.83524725895431671876534274190362e-03
            4.12357340313630305050907054464915e-01 1.75285319372739389898185891070170e-01 3.83524725895431671876534274190362e-03
            1.75285319372739389898185891070170e-01 4.12357340313630305050907054464915e-01 3.83524725895431671876534274190362e-03
            4.39048383245843498734473087097285e-01 4.39048383245843498734473087097285e-01 3.43814785469807697448896277592212e-03
            4.39048383245843498734473087097285e-01 1.21903233508313002531053825805429e-01 3.43814785469807697448896277592212e-03
            1.21903233508313002531053825805429e-01 4.39048383245843498734473087097285e-01 3.43814785469807697448896277592212e-03
            2.35052325458213695608122861813172e-01 2.35052325458213695608122861813172e-01 4.03399885831912195094384188109871e-03
            2.35052325458213695608122861813172e-01 5.29895349083572608783754276373656e-01 4.03399885831912195094384188109871e-03
            5.29895349083572608783754276373656e-01 2.35052325458213695608122861813172e-01 4.03399885831912195094384188109871e-03
            1.50821734779891525324302392618847e-01 1.50821734779891525324302392618847e-01 3.08553034072575962931672677314054e-03
            1.50821734779891525324302392618847e-01 6.98356530440216949351395214762306e-01 3.08553034072575962931672677314054e-03
            6.98356530440216949351395214762306e-01 1.50821734779891525324302392618847e-01 3.08553034072575962931672677314054e-03
            4.64273777634669981839721231153817e-01 4.64273777634669981839721231153817e-01 3.33061681062377188539125327793045e-03
            4.64273777634669981839721231153817e-01 7.14524447306600363205575376923662e-02 3.33061681062377188539125327793045e-03
            7.14524447306600363205575376923662e-02 4.64273777634669981839721231153817e-01 3.33061681062377188539125327793045e-03
            1.91161361877308921819462739222217e-01 1.91161361877308921819462739222217e-01 4.04661253784454298665362870224271e-03
            1.91161361877308921819462739222217e-01 6.17677276245382156361074521555565e-01 4.04661253784454298665362870224271e-03
            6.17677276245382156361074521555565e-01 1.91161361877308921819462739222217e-01 4.04661253784454298665362870224271e-03
            2.93500490934269150922197866293573e-02 2.93500490934269150922197866293573e-02 7.83711855575437081437850661558286e-04
            2.93500490934269150922197866293573e-02 9.41299901813146155937772618926829e-01 7.83711855575437081437850661558286e-04
            9.41299901813146155937772618926829e-01 2.93500490934269150922197866293573e-02 7.83711855575437081437850661558286e-04
            3.08145218991093261529101710038958e-01 3.08145218991093261529101710038958e-01 5.50413808085681386522702851493705e-03
            3.08145218991093261529101710038958e-01 3.83709562017813476941796579922084e-01 5.50413808085681386522702851493705e-03
            3.83709562017813476941796579922084e-01 3.08145218991093261529101710038958e-01 5.50413808085681386522702851493705e-03
            1.11178264788380914973586754967982e-01 1.11178264788380914973586754967982e-01 2.71177446206628640357205739519486e-03
            1.11178264788380914973586754967982e-01 7.77643470423238225563977721321862e-01 2.71177446206628640357205739519486e-03
            7.77643470423238225563977721321862e-01 1.11178264788380914973586754967982e-01 2.71177446206628640357205739519486e-03
            3.81002349603441869962949795080931e-01 3.81002349603441869962949795080931e-01 5.25317300340855629359193557093022e-03
            3.81002349603441869962949795080931e-01 2.37995300793116260074100409838138e-01 5.25317300340855629359193557093022e-03
            2.37995300793116260074100409838138e-01 3.81002349603441869962949795080931e-01 5.25317300340855629359193557093022e-03
            4.94480996747700407567549518716987e-01 4.94480996747700407567549518716987e-01 1.44795788173428361007843978569554e-03
            4.94480996747700407567549518716987e-01 1.10380065045991848649009625660256e-02 1.44795788173428361007843978569554e-03
            1.10380065045991848649009625660256e-02 4.94480996747700407567549518716987e-01 1.44795788173428361007843978569554e-03
            2.55002336075419500094341174190049e-01 3.22922639737693653594341469670326e-02 1.53084158353409063857419880605448e-03
            2.55002336075419500094341174190049e-01 7.12705399950811169240694198379060e-01 1.53084158353409063857419880605448e-03
            3.22922639737693653594341469670326e-02 2.55002336075419500094341174190049e-01 1.53084158353409063857419880605448e-03
            3.22922639737693653594341469670326e-02 7.12705399950811169240694198379060e-01 1.53084158353409063857419880605448e-03
            7.12705399950811169240694198379060e-01 2.55002336075419500094341174190049e-01 1.53084158353409063857419880605448e-03
            7.12705399950811169240694198379060e-01 3.22922639737693653594341469670326e-02 1.53084158353409063857419880605448e-03
            2.71121513603339969744610016277875e-01 9.41402098044081675487504412558337e-02 2.46163598764502456958669363018544e-03
            2.71121513603339969744610016277875e-01 6.34738276592251793317700503394008e-01 2.46163598764502456958669363018544e-03
            9.41402098044081675487504412558337e-02 2.71121513603339969744610016277875e-01 2.46163598764502456958669363018544e-03
            9.41402098044081675487504412558337e-02 6.34738276592251793317700503394008e-01 2.46163598764502456958669363018544e-03
            6.34738276592251793317700503394008e-01 2.71121513603339969744610016277875e-01 2.46163598764502456958669363018544e-03
            6.34738276592251793317700503394008e-01 9.41402098044081675487504412558337e-02 2.46163598764502456958669363018544e-03
            9.26886044848755363823045172466664e-02 6.49936721042402160142259504027606e-02 1.66186042169761695984353533361855e-03
            9.26886044848755363823045172466664e-02 8.42317723410884178214530493278289e-01 1.66186042169761695984353533361855e-03
            6.49936721042402160142259504027606e-02 9.26886044848755363823045172466664e-02 1.66186042169761695984353533361855e-03
            6.49936721042402160142259504027606e-02 8.42317723410884178214530493278289e-01 1.66186042169761695984353533361855e-03
            8.42317723410884178214530493278289e-01 9.26886044848755363823045172466664e-02 1.66186042169761695984353533361855e-03
            8.42317723410884178214530493278289e-01 6.49936721042402160142259504027606e-02 1.66186042169761695984353533361855e-03
            3.29454376584550512330196170296404e-01 7.37136644534187868149643918513902e-02 2.37136824651687809148437047213065e-03
            3.29454376584550512330196170296404e-01 5.96831958962030673099263822223293e-01 2.37136824651687809148437047213065e-03
            7.37136644534187868149643918513902e-02 3.29454376584550512330196170296404e-01 2.37136824651687809148437047213065e-03
            7.37136644534187868149643918513902e-02 5.96831958962030673099263822223293e-01 2.37136824651687809148437047213065e-03
            5.96831958962030673099263822223293e-01 3.29454376584550512330196170296404e-01 2.37136824651687809148437047213065e-03
            5.96831958962030673099263822223293e-01 7.37136644534187868149643918513902e-02 2.37136824651687809148437047213065e-03
            2.88378841130442042928194723572233e-01 5.00649881468919338844969502133608e-02 2.05690064632487509127534153208217e-03
            2.88378841130442042928194723572233e-01 6.61556170722666037065096134028863e-01 2.05690064632487509127534153208217e-03
            5.00649881468919338844969502133608e-02 2.88378841130442042928194723572233e-01 2.05690064632487509127534153208217e-03
            5.00649881468919338844969502133608e-02 6.61556170722666037065096134028863e-01 2.05690064632487509127534153208217e-03
            6.61556170722666037065096134028863e-01 2.88378841130442042928194723572233e-01 2.05690064632487509127534153208217e-03
            6.61556170722666037065096134028863e-01 5.00649881468919338844969502133608e-02 2.05690064632487509127534153208217e-03
            3.92172416405173862585797905921936e-01 6.58963384748946778834977067162981e-02 2.61767126144689344854898926939768e-03
            3.92172416405173862585797905921936e-01 5.41931245119931404019553156103939e-01 2.61767126144689344854898926939768e-03
            6.58963384748946778834977067162981e-02 3.92172416405173862585797905921936e-01 2.61767126144689344854898926939768e-03
            6.58963384748946778834977067162981e-02 5.41931245119931404019553156103939e-01 2.61767126144689344854898926939768e-03
            5.41931245119931404019553156103939e-01 3.92172416405173862585797905921936e-01 2.61767126144689344854898926939768e-03
            5.41931245119931404019553156103939e-01 6.58963384748946778834977067162981e-02 2.61767126144689344854898926939768e-03
            7.62225001558761768721828033790189e-03 1.23411852776327143031698252428896e-03 6.51699642263251529943068884875856e-05
            7.62225001558761768721828033790189e-03 9.91143631456649187860818983608624e-01 6.51699642263251529943068884875856e-05
            1.23411852776327143031698252428896e-03 7.62225001558761768721828033790189e-03 6.51699642263251529943068884875856e-05
            1.23411852776327143031698252428896e-03 9.91143631456649187860818983608624e-01 6.51699642263251529943068884875856e-05
            9.91143631456649187860818983608624e-01 7.62225001558761768721828033790189e-03 6.51699642263251529943068884875856e-05
            9.91143631456649187860818983608624e-01 1.23411852776327143031698252428896e-03 6.51699642263251529943068884875856e-05
            1.78207936627975682686297886903048e-01 1.05695827327609861989721196096070e-01 2.61385214196918104490041656617905e-03
            1.78207936627975682686297886903048e-01 7.16096236044414524712919956073165e-01 2.61385214196918104490041656617905e-03
            1.05695827327609861989721196096070e-01 1.78207936627975682686297886903048e-01 2.61385214196918104490041656617905e-03
            1.05695827327609861989721196096070e-01 7.16096236044414524712919956073165e-01 2.61385214196918104490041656617905e-03
            7.16096236044414524712919956073165e-01 1.78207936627975682686297886903048e-01 2.61385214196918104490041656617905e-03
            7.16096236044414524712919956073165e-01 1.05695827327609861989721196096070e-01 2.61385214196918104490041656617905e-03
            3.10248931459913612762591128557688e-01 1.23871670395550900756198586805112e-01 3.31556877077426826946893179126619e-03
            3.10248931459913612762591128557688e-01 5.65879398144535472603422476822743e-01 3.31556877077426826946893179126619e-03
            1.23871670395550900756198586805112e-01 3.10248931459913612762591128557688e-01 3.31556877077426826946893179126619e-03
            1.23871670395550900756198586805112e-01 5.65879398144535472603422476822743e-01 3.31556877077426826946893179126619e-03
            5.65879398144535472603422476822743e-01 3.10248931459913612762591128557688e-01 3.31556877077426826946893179126619e-03
            5.65879398144535472603422476822743e-01 1.23871670395550900756198586805112e-01 3.31556877077426826946893179126619e-03
            8.29157021331251853357713343939395e-02 2.80739761856614068735726874592729e-03 3.47617319858537106563173280093793e-04
            8.29157021331251853357713343939395e-02 9.14276900248308654894913161115255e-01 3.47617319858537106563173280093793e-04
            2.80739761856614068735726874592729e-03 8.29157021331251853357713343939395e-02 3.47617319858537106563173280093793e-04
            2.80739761856614068735726874592729e-03 9.14276900248308654894913161115255e-01 3.47617319858537106563173280093793e-04
            9.14276900248308654894913161115255e-01 8.29157021331251853357713343939395e-02 3.47617319858537106563173280093793e-04
            9.14276900248308654894913161115255e-01 2.80739761856614068735726874592729e-03 3.47617319858537106563173280093793e-04
            3.79638679346943119785606768346042e-01 1.10689778326021240340004681002029e-01 3.45918624369627664486004370303363e-03
            3.79638679346943119785606768346042e-01 5.09671542327035709263327589724213e-01 3.45918624369627664486004370303363e-03
            1.10689778326021240340004681002029e-01 3.79638679346943119785606768346042e-01 3.45918624369627664486004370303363e-03
            1.10689778326021240340004681002029e-01 5.09671542327035709263327589724213e-01 3.45918624369627664486004370303363e-03
            5.09671542327035709263327589724213e-01 3.79638679346943119785606768346042e-01 3.45918624369627664486004370303363e-03
            5.09671542327035709263327589724213e-01 1.10689778326021240340004681002029e-01 3.45918624369627664486004370303363e-03
            1.98068292114543353932276659179479e-01 2.75642261370565543054933499433901e-03 5.08853424677458213154501720509870e-04
            1.98068292114543353932276659179479e-01 7.99175285271750968085768818127690e-01 5.08853424677458213154501720509870e-04
            2.75642261370565543054933499433901e-03 1.98068292114543353932276659179479e-01 5.08853424677458213154501720509870e-04
            2.75642261370565543054933499433901e-03 7.99175285271750968085768818127690e-01 5.08853424677458213154501720509870e-04
            7.99175285271750968085768818127690e-01 1.98068292114543353932276659179479e-01 5.08853424677458213154501720509870e-04
            7.99175285271750968085768818127690e-01 2.75642261370565543054933499433901e-03 5.08853424677458213154501720509870e-04
            1.47947945322073032281551263622532e-01 7.20516990960159003920537656995293e-02 2.26549564853226886862458400173637e-03
            1.47947945322073032281551263622532e-01 7.80000355581911053448607162863482e-01 2.26549564853226886862458400173637e-03
            7.20516990960159003920537656995293e-02 1.47947945322073032281551263622532e-01 2.26549564853226886862458400173637e-03
            7.20516990960159003920537656995293e-02 7.80000355581911053448607162863482e-01 2.26549564853226886862458400173637e-03
            7.80000355581911053448607162863482e-01 1.47947945322073032281551263622532e-01 2.26549564853226886862458400173637e-03
            7.80000355581911053448607162863482e-01 7.20516990960159003920537656995293e-02 2.26549564853226886862458400173637e-03
            2.29202766553467329746496261577704e-01 1.44286115706842923978037163124100e-02 1.18964024781928489650495439633460e-03
            2.29202766553467329746496261577704e-01 7.56368621875848412550169541646028e-01 1.18964024781928489650495439633460e-03
            1.44286115706842923978037163124100e-02 2.29202766553467329746496261577704e-01 1.18964024781928489650495439633460e-03
            1.44286115706842923978037163124100e-02 7.56368621875848412550169541646028e-01 1.18964024781928489650495439633460e-03
            7.56368621875848412550169541646028e-01 2.29202766553467329746496261577704e-01 1.18964024781928489650495439633460e-03
            7.56368621875848412550169541646028e-01 1.44286115706842923978037163124100e-02 1.18964024781928489650495439633460e-03
            3.57473850344384502797367986204335e-01 2.74256669741186880767669364900030e-03 6.08439554852720633319751453171875e-04
            3.57473850344384502797367986204335e-01 6.39783582958203589363677110668505e-01 6.08439554852720633319751453171875e-04
            2.74256669741186880767669364900030e-03 3.57473850344384502797367986204335e-01 6.08439554852720633319751453171875e-04
            2.74256669741186880767669364900030e-03 6.39783582958203589363677110668505e-01 6.08439554852720633319751453171875e-04
            6.39783582958203589363677110668505e-01 3.57473850344384502797367986204335e-01 6.08439554852720633319751453171875e-04
            6.39783582958203589363677110668505e-01 2.74256669741186880767669364900030e-03 6.08439554852720633319751453171875e-04
            1.25249672229437725157197291991906e-01 3.90243298422159254346119894307776e-02 1.55651362018915527703211143517592e-03
            1.25249672229437725157197291991906e-01 8.35725997928346342469296814670088e-01 1.55651362018915527703211143517592e-03
            3.90243298422159254346119894307776e-02 1.25249672229437725157197291991906e-01 1.55651362018915527703211143517592e-03
            3.90243298422159254346119894307776e-02 8.35725997928346342469296814670088e-01 1.55651362018915527703211143517592e-03
            8.35725997928346342469296814670088e-01 1.25249672229437725157197291991906e-01 1.55651362018915527703211143517592e-03
            8.35725997928346342469296814670088e-01 3.90243298422159254346119894307776e-02 1.55651362018915527703211143517592e-03
            2.09805646004062387732780337046279e-02 7.47466346593332695708022939129478e-03 3.24092666406562907640415138743606e-04
            2.09805646004062387732780337046279e-02 9.71544771933660511464836417871993e-01 3.24092666406562907640415138743606e-04
            7.47466346593332695708022939129478e-03 2.09805646004062387732780337046279e-02 3.24092666406562907640415138743606e-04
            7.47466346593332695708022939129478e-03 9.71544771933660511464836417871993e-01 3.24092666406562907640415138743606e-04
            9.71544771933660511464836417871993e-01 2.09805646004062387732780337046279e-02 3.24092666406562907640415138743606e-04
            9.71544771933660511464836417871993e-01 7.47466346593332695708022939129478e-03 3.24092666406562907640415138743606e-04
            2.69131611744009735520677395470557e-01 1.75571778601519262741348370582273e-01 4.17258237302728460932366161273421e-03
            2.69131611744009735520677395470557e-01 5.55296609654471029493549849576084e-01 4.17258237302728460932366161273421e-03
            1.75571778601519262741348370582273e-01 2.69131611744009735520677395470557e-01 4.17258237302728460932366161273421e-03
            1.75571778601519262741348370582273e-01 5.55296609654471029493549849576084e-01 4.17258237302728460932366161273421e-03
            5.55296609654471029493549849576084e-01 2.69131611744009735520677395470557e-01 4.17258237302728460932366161273421e-03
            5.55296609654471029493549849576084e-01 1.75571778601519262741348370582273e-01 4.17258237302728460932366161273421e-03
            3.46219872421893559710781573812710e-01 1.71226758850752919238757954190078e-01 4.45190142462590764810270727025454e-03
            3.46219872421893559710781573812710e-01 4.82553368727353548806036087626126e-01 4.45190142462590764810270727025454e-03
            1.71226758850752919238757954190078e-01 3.46219872421893559710781573812710e-01 4.45190142462590764810270727025454e-03
            1.71226758850752919238757954190078e-01 4.82553368727353548806036087626126e-01 4.45190142462590764810270727025454e-03
            4.82553368727353548806036087626126e-01 3.46219872421893559710781573812710e-01 4.45190142462590764810270727025454e-03
            4.82553368727353548806036087626126e-01 1.71226758850752919238757954190078e-01 4.45190142462590764810270727025454e-03
            4.42387742658068516821145976791740e-01 3.43061434540936835424496109681058e-02 2.38190698469414093713103497407246e-03
            4.42387742658068516821145976791740e-01 5.23306113887837875964237355219666e-01 2.38190698469414093713103497407246e-03
            3.43061434540936835424496109681058e-02 4.42387742658068516821145976791740e-01 2.38190698469414093713103497407246e-03
            3.43061434540936835424496109681058e-02 5.23306113887837875964237355219666e-01 2.38190698469414093713103497407246e-03
            5.23306113887837875964237355219666e-01 4.42387742658068516821145976791740e-01 2.38190698469414093713103497407246e-03
            5.23306113887837875964237355219666e-01 3.43061434540936835424496109681058e-02 2.38190698469414093713103497407246e-03
            2.74633883722658844739328287687385e-01 3.02963551184710737362859589438813e-03 6.16571745718724620669015656915235e-04
            2.74633883722658844739328287687385e-01 7.22336480765494060030107448255876e-01 6.16571745718724620669015656915235e-04
            3.02963551184710737362859589438813e-03 2.74633883722658844739328287687385e-01 6.16571745718724620669015656915235e-04
            3.02963551184710737362859589438813e-03 7.22336480765494060030107448255876e-01 6.16571745718724620669015656915235e-04
            7.22336480765494060030107448255876e-01 2.74633883722658844739328287687385e-01 6.16571745718724620669015656915235e-04
            7.22336480765494060030107448255876e-01 3.02963551184710737362859589438813e-03 6.16571745718724620669015656915235e-04
            2.23701284655144105872892623665393e-01 7.24869584556441376443913782168238e-02 2.64051811768303964708670683592118e-03
            2.23701284655144105872892623665393e-01 7.03811756889211825871655037190067e-01 2.64051811768303964708670683592118e-03
            7.24869584556441376443913782168238e-02 2.23701284655144105872892623665393e-01 2.64051811768303964708670683592118e-03
            7.24869584556441376443913782168238e-02 7.03811756889211825871655037190067e-01 2.64051811768303964708670683592118e-03
            7.03811756889211825871655037190067e-01 2.23701284655144105872892623665393e-01 2.64051811768303964708670683592118e-03
            7.03811756889211825871655037190067e-01 7.24869584556441376443913782168238e-02 2.64051811768303964708670683592118e-03
            2.26472947511629774108854462610907e-01 1.33817023340663948660633764120576e-01 3.60197908152549116367335813038153e-03
            2.26472947511629774108854462610907e-01 6.39710029147706249474936157639604e-01 3.60197908152549116367335813038153e-03
            1.33817023340663948660633764120576e-01 2.26472947511629774108854462610907e-01 3.60197908152549116367335813038153e-03
            1.33817023340663948660633764120576e-01 6.39710029147706249474936157639604e-01 3.60197908152549116367335813038153e-03
            6.39710029147706249474936157639604e-01 2.26472947511629774108854462610907e-01 3.60197908152549116367335813038153e-03
            6.39710029147706249474936157639604e-01 1.33817023340663948660633764120576e-01 3.60197908152549116367335813038153e-03
            1.63361966829370430787093937397003e-01 1.76742202382502745783288133907263e-02 1.27502323317565295981990125540051e-03
            1.63361966829370430787093937397003e-01 8.18963812932379298104024201165885e-01 1.27502323317565295981990125540051e-03
            1.76742202382502745783288133907263e-02 1.63361966829370430787093937397003e-01 1.27502323317565295981990125540051e-03
            1.76742202382502745783288133907263e-02 8.18963812932379298104024201165885e-01 1.27502323317565295981990125540051e-03
            8.18963812932379298104024201165885e-01 1.63361966829370430787093937397003e-01 1.27502323317565295981990125540051e-03
            8.18963812932379298104024201165885e-01 1.76742202382502745783288133907263e-02 1.27502323317565295981990125540051e-03
            1.90791272854378857193324847685290e-01 4.22082170395821076880338296177797e-02 2.04656057870133056483807898473515e-03
            1.90791272854378857193324847685290e-01 7.67000510106039090629792553954758e-01 2.04656057870133056483807898473515e-03
            4.22082170395821076880338296177797e-02 1.90791272854378857193324847685290e-01 2.04656057870133056483807898473515e-03
            4.22082170395821076880338296177797e-02 7.67000510106039090629792553954758e-01 2.04656057870133056483807898473515e-03
            7.67000510106039090629792553954758e-01 1.90791272854378857193324847685290e-01 2.04656057870133056483807898473515e-03
            7.67000510106039090629792553954758e-01 4.22082170395821076880338296177797e-02 2.04656057870133056483807898473515e-03
            3.58165870631430682813345356407808e-01 3.44163203969896150380414212577307e-02 2.23469587180631949416276249564817e-03
            3.58165870631430682813345356407808e-01 6.07417808971579709087507126241690e-01 2.23469587180631949416276249564817e-03
            3.44163203969896150380414212577307e-02 3.58165870631430682813345356407808e-01 2.23469587180631949416276249564817e-03
            3.44163203969896150380414212577307e-02 6.07417808971579709087507126241690e-01 2.23469587180631949416276249564817e-03
            6.07417808971579709087507126241690e-01 3.58165870631430682813345356407808e-01 2.23469587180631949416276249564817e-03
            6.07417808971579709087507126241690e-01 3.44163203969896150380414212577307e-02 2.23469587180631949416276249564817e-03
            9.93849892050521904751647639386647e-02 1.74529902610205088497874470476745e-02 1.05061121151762428514719882599593e-03
            9.93849892050521904751647639386647e-02 8.83162020533927227816661797987763e-01 1.05061121151762428514719882599593e-03
            1.74529902610205088497874470476745e-02 9.93849892050521904751647639386647e-02 1.05061121151762428514719882599593e-03
            1.74529902610205088497874470476745e-02 8.83162020533927227816661797987763e-01 1.05061121151762428514719882599593e-03
            8.83162020533927227816661797987763e-01 9.93849892050521904751647639386647e-02 1.05061121151762428514719882599593e-03
            8.83162020533927227816661797987763e-01 1.74529902610205088497874470476745e-02 1.05061121151762428514719882599593e-03
            4.50232897891652383837168827085407e-01 1.22783484472307590634754248526406e-03 4.11044553657110396053392387116787e-04
            4.50232897891652383837168827085407e-01 5.48539267263624563675250556116225e-01 4.11044553657110396053392387116787e-04
            1.22783484472307590634754248526406e-03 4.50232897891652383837168827085407e-01 4.11044553657110396053392387116787e-04
            1.22783484472307590634754248526406e-03 5.48539267263624563675250556116225e-01 4.11044553657110396053392387116787e-04
            5.48539267263624563675250556116225e-01 4.50232897891652383837168827085407e-01 4.11044553657110396053392387116787e-04
            5.48539267263624563675250556116225e-01 1.22783484472307590634754248526406e-03 4.11044553657110396053392387116787e-04
            3.04860292986679271010075353842694e-01 2.36263190066308664816219220483617e-01 5.10667280776457097740683721553978e-03
            3.04860292986679271010075353842694e-01 4.58876516947012036418129810044775e-01 5.10667280776457097740683721553978e-03
            2.36263190066308664816219220483617e-01 3.04860292986679271010075353842694e-01 5.10667280776457097740683721553978e-03
            2.36263190066308664816219220483617e-01 4.58876516947012036418129810044775e-01 5.10667280776457097740683721553978e-03
            4.58876516947012036418129810044775e-01 3.04860292986679271010075353842694e-01 5.10667280776457097740683721553978e-03
            4.58876516947012036418129810044775e-01 2.36263190066308664816219220483617e-01 5.10667280776457097740683721553978e-03
            3.13583606270695147699711924360599e-01 1.55229285287317442243448795125005e-02 1.52011067741504861568213602396327e-03
            3.13583606270695147699711924360599e-01 6.70893465200573135831518811755814e-01 1.52011067741504861568213602396327e-03
            1.55229285287317442243448795125005e-02 3.13583606270695147699711924360599e-01 1.52011067741504861568213602396327e-03
            1.55229285287317442243448795125005e-02 6.70893465200573135831518811755814e-01 1.52011067741504861568213602396327e-03
            6.70893465200573135831518811755814e-01 3.13583606270695147699711924360599e-01 1.52011067741504861568213602396327e-03
            6.70893465200573135831518811755814e-01 1.55229285287317442243448795125005e-02 1.52011067741504861568213602396327e-03
            1.34806288266682894327885833263281e-01 3.84804577084505751438658904817203e-03 5.54805076808530576136513356999558e-04
            1.34806288266682894327885833263281e-01 8.61345665962471973564618110685842e-01 5.54805076808530576136513356999558e-04
            3.84804577084505751438658904817203e-03 1.34806288266682894327885833263281e-01 5.54805076808530576136513356999558e-04
            3.84804577084505751438658904817203e-03 8.61345665962471973564618110685842e-01 5.54805076808530576136513356999558e-04
            8.61345665962471973564618110685842e-01 1.34806288266682894327885833263281e-01 5.54805076808530576136513356999558e-04
            8.61345665962471973564618110685842e-01 3.84804577084505751438658904817203e-03 5.54805076808530576136513356999558e-04
            6.58281924328198569940084894369647e-02 3.80724693230130931120847037618660e-02 1.33345058300172688604823267866095e-03
            6.58281924328198569940084894369647e-02 8.96099338244166987443861671636114e-01 1.33345058300172688604823267866095e-03
            3.80724693230130931120847037618660e-02 6.58281924328198569940084894369647e-02 1.33345058300172688604823267866095e-03
            3.80724693230130931120847037618660e-02 8.96099338244166987443861671636114e-01 1.33345058300172688604823267866095e-03
            8.96099338244166987443861671636114e-01 6.58281924328198569940084894369647e-02 1.33345058300172688604823267866095e-03
            8.96099338244166987443861671636114e-01 3.80724693230130931120847037618660e-02 1.33345058300172688604823267866095e-03
            5.31928439194577426518684148959437e-02 1.21702541539402027404026540580162e-02 6.67804382032510693116555966497572e-04
            5.31928439194577426518684148959437e-02 9.34636901926601981749342940020142e-01 6.67804382032510693116555966497572e-04
            1.21702541539402027404026540580162e-02 5.31928439194577426518684148959437e-02 6.67804382032510693116555966497572e-04
            1.21702541539402027404026540580162e-02 9.34636901926601981749342940020142e-01 6.67804382032510693116555966497572e-04
            9.34636901926601981749342940020142e-01 5.31928439194577426518684148959437e-02 6.67804382032510693116555966497572e-04
            9.34636901926601981749342940020142e-01 1.21702541539402027404026540580162e-02 6.67804382032510693116555966497572e-04
            4.06479626359585802242158081298840e-02 8.58559044683948331504186324281136e-04 1.33088911946721138177332499630268e-04
            4.06479626359585802242158081298840e-02 9.58493478319357494754626713984180e-01 1.33088911946721138177332499630268e-04
            8.58559044683948331504186324281136e-04 4.06479626359585802242158081298840e-02 1.33088911946721138177332499630268e-04
            8.58559044683948331504186324281136e-04 9.58493478319357494754626713984180e-01 1.33088911946721138177332499630268e-04
            9.58493478319357494754626713984180e-01 4.06479626359585802242158081298840e-02 1.33088911946721138177332499630268e-04
            9.58493478319357494754626713984180e-01 8.58559044683948331504186324281136e-04 1.33088911946721138177332499630268e-04
            4.06447567212284865956206658665906e-01 1.23358378103272329356920877785342e-02 1.44384349566076155939320369014922e-03
            4.06447567212284865956206658665906e-01 5.81216594977387868148355209996225e-01 1.44384349566076155939320369014922e-03
            1.23358378103272329356920877785342e-02 4.06447567212284865956206658665906e-01 1.44384349566076155939320369014922e-03
            1.23358378103272329356920877785342e-02 5.81216594977387868148355209996225e-01 1.44384349566076155939320369014922e-03
            5.81216594977387868148355209996225e-01 4.06447567212284865956206658665906e-01 1.44384349566076155939320369014922e-03
            5.81216594977387868148355209996225e-01 1.23358378103272329356920877785342e-02 1.44384349566076155939320369014922e-03
            ];
        
    case  37
        
        % ALG. DEG.:   37
        % PTS CARD.:  252
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.0e-15
        
        xyw=[
            1.02303932157760305021731994656875e-01 1.02303932157760305021731994656875e-01 7.56994885929007654823408302036114e-04
            1.02303932157760305021731994656875e-01 7.95392135684479417712111626315163e-01 7.56994885929007654823408302036114e-04
            7.95392135684479417712111626315163e-01 1.02303932157760305021731994656875e-01 7.56994885929007654823408302036114e-04
            3.50202358754684098252063506606646e-02 3.50202358754684098252063506606646e-02 5.52341834522307299250309942095782e-04
            3.50202358754684098252063506606646e-02 9.29959528249063138716223875235301e-01 5.52341834522307299250309942095782e-04
            9.29959528249063138716223875235301e-01 3.50202358754684098252063506606646e-02 5.52341834522307299250309942095782e-04
            2.59243490456549541267605718530831e-01 2.59243490456549541267605718530831e-01 3.79254617818526248984589699375647e-03
            2.59243490456549541267605718530831e-01 4.81513019086900917464788562938338e-01 3.79254617818526248984589699375647e-03
            4.81513019086900917464788562938338e-01 2.59243490456549541267605718530831e-01 3.79254617818526248984589699375647e-03
            1.39441235872292679243145130385528e-01 1.39441235872292679243145130385528e-01 2.62503007716438857901963821461777e-03
            1.39441235872292679243145130385528e-01 7.21117528255414641513709739228943e-01 2.62503007716438857901963821461777e-03
            7.21117528255414641513709739228943e-01 1.39441235872292679243145130385528e-01 2.62503007716438857901963821461777e-03
            4.61168458890810040173846573452465e-01 4.61168458890810040173846573452465e-01 2.94041733963891450606698008130024e-03
            4.61168458890810040173846573452465e-01 7.76630822183799196523068530950695e-02 2.94041733963891450606698008130024e-03
            7.76630822183799196523068530950695e-02 4.61168458890810040173846573452465e-01 2.94041733963891450606698008130024e-03
            3.79153717955037827813669082388515e-01 3.79153717955037827813669082388515e-01 4.32227828704266625603480633799336e-03
            3.79153717955037827813669082388515e-01 2.41692564089924344372661835222971e-01 4.32227828704266625603480633799336e-03
            2.41692564089924344372661835222971e-01 3.79153717955037827813669082388515e-01 4.32227828704266625603480633799336e-03
            1.73411753639024328321749379711036e-02 1.73411753639024328321749379711036e-02 4.38793626075407376500858713797015e-04
            1.73411753639024328321749379711036e-02 9.65317649272195099641180604521651e-01 4.38793626075407376500858713797015e-04
            9.65317649272195099641180604521651e-01 1.73411753639024328321749379711036e-02 4.38793626075407376500858713797015e-04
            7.79889301806426238350411495048320e-02 7.79889301806426238350411495048320e-02 1.79311680137484233399369770722842e-03
            7.79889301806426238350411495048320e-02 8.44022139638714752329917700990336e-01 1.79311680137484233399369770722842e-03
            8.44022139638714752329917700990336e-01 7.79889301806426238350411495048320e-02 1.79311680137484233399369770722842e-03
            4.79244935556920326202856585950940e-01 4.79244935556920326202856585950940e-01 2.49962141423790921296310685306707e-03
            4.79244935556920326202856585950940e-01 4.15101288861593475942868280981202e-02 2.49962141423790921296310685306707e-03
            4.15101288861593475942868280981202e-02 4.79244935556920326202856585950940e-01 2.49962141423790921296310685306707e-03
            4.12061405479689502051598992693471e-01 4.12061405479689502051598992693471e-01 4.53942125276982424986194075700041e-03
            4.12061405479689502051598992693471e-01 1.75877189040620995896802014613058e-01 4.53942125276982424986194075700041e-03
            1.75877189040620995896802014613058e-01 4.12061405479689502051598992693471e-01 4.53942125276982424986194075700041e-03
            4.98410574413956974915151931782020e-01 4.98410574413956974915151931782020e-01 7.07839054597255682051148983902067e-04
            4.98410574413956974915151931782020e-01 3.17885117208605016969613643595949e-03 7.07839054597255682051148983902067e-04
            3.17885117208605016969613643595949e-03 4.98410574413956974915151931782020e-01 7.07839054597255682051148983902067e-04
            3.09125948685350482314504461101023e-01 3.09125948685350482314504461101023e-01 5.10331832458679572617210595808501e-03
            3.09125948685350482314504461101023e-01 3.81748102629299035370991077797953e-01 5.10331832458679572617210595808501e-03
            3.81748102629299035370991077797953e-01 3.09125948685350482314504461101023e-01 5.10331832458679572617210595808501e-03
            3.67156682198311505940790944180208e-03 3.67156682198311505940790944180208e-03 8.66476678904295929527917818901983e-05
            3.67156682198311505940790944180208e-03 9.92656866356033784626333726919256e-01 8.66476678904295929527917818901983e-05
            9.92656866356033784626333726919256e-01 3.67156682198311505940790944180208e-03 8.66476678904295929527917818901983e-05
            1.89364261200196171142096090989071e-01 1.89364261200196171142096090989071e-01 4.14530662852686040964478308978869e-03
            1.89364261200196171142096090989071e-01 6.21271477599607657715807818021858e-01 4.14530662852686040964478308978869e-03
            6.21271477599607657715807818021858e-01 1.89364261200196171142096090989071e-01 4.14530662852686040964478308978869e-03
            3.16024096496639927966754157750984e-01 3.38290906827213161411904707165377e-02 1.04959604867473662963284208871073e-03
            3.16024096496639927966754157750984e-01 6.50146812820638797525418794975849e-01 1.04959604867473662963284208871073e-03
            3.38290906827213161411904707165377e-02 3.16024096496639927966754157750984e-01 1.04959604867473662963284208871073e-03
            3.38290906827213161411904707165377e-02 6.50146812820638797525418794975849e-01 1.04959604867473662963284208871073e-03
            6.50146812820638797525418794975849e-01 3.16024096496639927966754157750984e-01 1.04959604867473662963284208871073e-03
            6.50146812820638797525418794975849e-01 3.38290906827213161411904707165377e-02 1.04959604867473662963284208871073e-03
            1.05359411960281923459348263349966e-01 3.31459895561884293480048313540465e-02 9.61312586091038847636858211131994e-04
            1.05359411960281923459348263349966e-01 8.61494598483529716581585944368271e-01 9.61312586091038847636858211131994e-04
            3.31459895561884293480048313540465e-02 1.05359411960281923459348263349966e-01 9.61312586091038847636858211131994e-04
            3.31459895561884293480048313540465e-02 8.61494598483529716581585944368271e-01 9.61312586091038847636858211131994e-04
            8.61494598483529716581585944368271e-01 1.05359411960281923459348263349966e-01 9.61312586091038847636858211131994e-04
            8.61494598483529716581585944368271e-01 3.31459895561884293480048313540465e-02 9.61312586091038847636858211131994e-04
            7.49810217841138243466403423553857e-02 2.03347607611732879295818321452316e-03 2.40221505000938354099787597206728e-04
            7.49810217841138243466403423553857e-02 9.22985502139768820839549334777985e-01 2.40221505000938354099787597206728e-04
            2.03347607611732879295818321452316e-03 7.49810217841138243466403423553857e-02 2.40221505000938354099787597206728e-04
            2.03347607611732879295818321452316e-03 9.22985502139768820839549334777985e-01 2.40221505000938354099787597206728e-04
            9.22985502139768820839549334777985e-01 7.49810217841138243466403423553857e-02 2.40221505000938354099787597206728e-04
            9.22985502139768820839549334777985e-01 2.03347607611732879295818321452316e-03 2.40221505000938354099787597206728e-04
            1.19486593380355193838582295029482e-01 5.55104031624588586946167367841554e-02 1.51627373004474563268628717338515e-03
            1.19486593380355193838582295029482e-01 8.25003003457186023794633911165874e-01 1.51627373004474563268628717338515e-03
            5.55104031624588586946167367841554e-02 1.19486593380355193838582295029482e-01 1.51627373004474563268628717338515e-03
            5.55104031624588586946167367841554e-02 8.25003003457186023794633911165874e-01 1.51627373004474563268628717338515e-03
            8.25003003457186023794633911165874e-01 1.19486593380355193838582295029482e-01 1.51627373004474563268628717338515e-03
            8.25003003457186023794633911165874e-01 5.55104031624588586946167367841554e-02 1.51627373004474563268628717338515e-03
            3.19235165045989144338989262905670e-01 2.38060847117024854124878174843616e-01 3.65945316462084864819748197817262e-03
            3.19235165045989144338989262905670e-01 4.42703987836986001536132562250714e-01 3.65945316462084864819748197817262e-03
            2.38060847117024854124878174843616e-01 3.19235165045989144338989262905670e-01 3.65945316462084864819748197817262e-03
            2.38060847117024854124878174843616e-01 4.42703987836986001536132562250714e-01 3.65945316462084864819748197817262e-03
            4.42703987836986001536132562250714e-01 3.19235165045989144338989262905670e-01 3.65945316462084864819748197817262e-03
            4.42703987836986001536132562250714e-01 2.38060847117024854124878174843616e-01 3.65945316462084864819748197817262e-03
            3.91788257723670096943635599018307e-01 7.52705489375098402149433240992948e-02 2.69415942149049117068648762085559e-03
            3.91788257723670096943635599018307e-01 5.32941193338820062841421076882398e-01 2.69415942149049117068648762085559e-03
            7.52705489375098402149433240992948e-02 3.91788257723670096943635599018307e-01 2.69415942149049117068648762085559e-03
            7.52705489375098402149433240992948e-02 5.32941193338820062841421076882398e-01 2.69415942149049117068648762085559e-03
            5.32941193338820062841421076882398e-01 3.91788257723670096943635599018307e-01 2.69415942149049117068648762085559e-03
            5.32941193338820062841421076882398e-01 7.52705489375098402149433240992948e-02 2.69415942149049117068648762085559e-03
            3.17543407214198269983285172202159e-01 5.36208316291131165121086610270140e-02 2.28460795949074089647434782079927e-03
            3.17543407214198269983285172202159e-01 6.28835761156688599626818358956371e-01 2.28460795949074089647434782079927e-03
            5.36208316291131165121086610270140e-02 3.17543407214198269983285172202159e-01 2.28460795949074089647434782079927e-03
            5.36208316291131165121086610270140e-02 6.28835761156688599626818358956371e-01 2.28460795949074089647434782079927e-03
            6.28835761156688599626818358956371e-01 3.17543407214198269983285172202159e-01 2.28460795949074089647434782079927e-03
            6.28835761156688599626818358956371e-01 5.36208316291131165121086610270140e-02 2.28460795949074089647434782079927e-03
            3.26764524592731753216412471374497e-01 9.12947670950428336045234800621984e-02 2.86017995322923818460325584567272e-03
            3.26764524592731753216412471374497e-01 5.81940708312225440934639664192218e-01 2.86017995322923818460325584567272e-03
            9.12947670950428336045234800621984e-02 3.26764524592731753216412471374497e-01 2.86017995322923818460325584567272e-03
            9.12947670950428336045234800621984e-02 5.81940708312225440934639664192218e-01 2.86017995322923818460325584567272e-03
            5.81940708312225440934639664192218e-01 3.26764524592731753216412471374497e-01 2.86017995322923818460325584567272e-03
            5.81940708312225440934639664192218e-01 9.12947670950428336045234800621984e-02 2.86017995322923818460325584567272e-03
            3.18749666827180144768050240600132e-01 1.35115749070284674182573780854000e-01 3.35600242454283520521229888800008e-03
            3.18749666827180144768050240600132e-01 5.46134584102535236560527209803695e-01 3.35600242454283520521229888800008e-03
            1.35115749070284674182573780854000e-01 3.18749666827180144768050240600132e-01 3.35600242454283520521229888800008e-03
            1.35115749070284674182573780854000e-01 5.46134584102535236560527209803695e-01 3.35600242454283520521229888800008e-03
            5.46134584102535236560527209803695e-01 3.18749666827180144768050240600132e-01 3.35600242454283520521229888800008e-03
            5.46134584102535236560527209803695e-01 1.35115749070284674182573780854000e-01 3.35600242454283520521229888800008e-03
            6.74578790813868262032926281790424e-02 4.07701237566786764410231569399912e-02 1.15853875465657374897732179164223e-03
            6.74578790813868262032926281790424e-02 8.91771997161934559805729350046022e-01 1.15853875465657374897732179164223e-03
            4.07701237566786764410231569399912e-02 6.74578790813868262032926281790424e-02 1.15853875465657374897732179164223e-03
            4.07701237566786764410231569399912e-02 8.91771997161934559805729350046022e-01 1.15853875465657374897732179164223e-03
            8.91771997161934559805729350046022e-01 6.74578790813868262032926281790424e-02 1.15853875465657374897732179164223e-03
            8.91771997161934559805729350046022e-01 4.07701237566786764410231569399912e-02 1.15853875465657374897732179164223e-03
            1.83794409210811804378238321078243e-02 2.99027787656391775758524254058557e-03 1.54334630985266210191630964487786e-04
            1.83794409210811804378238321078243e-02 9.78630281202354956882061287615215e-01 1.54334630985266210191630964487786e-04
            2.99027787656391775758524254058557e-03 1.83794409210811804378238321078243e-02 1.54334630985266210191630964487786e-04
            2.99027787656391775758524254058557e-03 9.78630281202354956882061287615215e-01 1.54334630985266210191630964487786e-04
            9.78630281202354956882061287615215e-01 1.83794409210811804378238321078243e-02 1.54334630985266210191630964487786e-04
            9.78630281202354956882061287615215e-01 2.99027787656391775758524254058557e-03 1.54334630985266210191630964487786e-04
            4.76569358622159489069503024438745e-02 1.72509604538075254909212929987916e-02 6.02351384720529949186318408038687e-04
            4.76569358622159489069503024438745e-02 9.35092103683976549888257068232633e-01 6.02351384720529949186318408038687e-04
            1.72509604538075254909212929987916e-02 4.76569358622159489069503024438745e-02 6.02351384720529949186318408038687e-04
            1.72509604538075254909212929987916e-02 9.35092103683976549888257068232633e-01 6.02351384720529949186318408038687e-04
            9.35092103683976549888257068232633e-01 4.76569358622159489069503024438745e-02 6.02351384720529949186318408038687e-04
            9.35092103683976549888257068232633e-01 1.72509604538075254909212929987916e-02 6.02351384720529949186318408038687e-04
            3.35722787038010062055093385424698e-01 1.81116409195531985476534941881255e-01 4.08187312607179424855941007876936e-03
            3.35722787038010062055093385424698e-01 4.83160803766457980223947288322961e-01 4.08187312607179424855941007876936e-03
            1.81116409195531985476534941881255e-01 3.35722787038010062055093385424698e-01 4.08187312607179424855941007876936e-03
            1.81116409195531985476534941881255e-01 4.83160803766457980223947288322961e-01 4.08187312607179424855941007876936e-03
            4.83160803766457980223947288322961e-01 3.35722787038010062055093385424698e-01 4.08187312607179424855941007876936e-03
            4.83160803766457980223947288322961e-01 1.81116409195531985476534941881255e-01 4.08187312607179424855941007876936e-03
            1.35671607916117625824981018922699e-01 9.01302718482900888652409321366576e-02 2.18226201409495428745755063459910e-03
            1.35671607916117625824981018922699e-01 7.74198120235592313065353664569557e-01 2.18226201409495428745755063459910e-03
            9.01302718482900888652409321366576e-02 1.35671607916117625824981018922699e-01 2.18226201409495428745755063459910e-03
            9.01302718482900888652409321366576e-02 7.74198120235592313065353664569557e-01 2.18226201409495428745755063459910e-03
            7.74198120235592313065353664569557e-01 1.35671607916117625824981018922699e-01 2.18226201409495428745755063459910e-03
            7.74198120235592313065353664569557e-01 9.01302718482900888652409321366576e-02 2.18226201409495428745755063459910e-03
            4.17693614359697928950510004142416e-02 4.01709708723125332591186165132058e-03 2.74898275323659198928621938406991e-04
            4.17693614359697928950510004142416e-02 9.54213541476798932094993688224349e-01 2.74898275323659198928621938406991e-04
            4.01709708723125332591186165132058e-03 4.17693614359697928950510004142416e-02 2.74898275323659198928621938406991e-04
            4.01709708723125332591186165132058e-03 9.54213541476798932094993688224349e-01 2.74898275323659198928621938406991e-04
            9.54213541476798932094993688224349e-01 4.17693614359697928950510004142416e-02 2.74898275323659198928621938406991e-04
            9.54213541476798932094993688224349e-01 4.01709708723125332591186165132058e-03 2.74898275323659198928621938406991e-04
            1.91263737438076231178740727045806e-01 8.09653202292540813411747535610630e-02 2.62321907934608654355823631476596e-03
            1.91263737438076231178740727045806e-01 7.27770942332669701357872327207588e-01 2.62321907934608654355823631476596e-03
            8.09653202292540813411747535610630e-02 1.91263737438076231178740727045806e-01 2.62321907934608654355823631476596e-03
            8.09653202292540813411747535610630e-02 7.27770942332669701357872327207588e-01 2.62321907934608654355823631476596e-03
            7.27770942332669701357872327207588e-01 1.91263737438076231178740727045806e-01 2.62321907934608654355823631476596e-03
            7.27770942332669701357872327207588e-01 8.09653202292540813411747535610630e-02 2.62321907934608654355823631476596e-03
            4.48920329956642827351487312625977e-01 1.68814585101307729209096919475996e-02 1.62924472819564610491516187806837e-03
            4.48920329956642827351487312625977e-01 5.34198211533226330338663956354139e-01 1.62924472819564610491516187806837e-03
            1.68814585101307729209096919475996e-02 4.48920329956642827351487312625977e-01 1.62924472819564610491516187806837e-03
            1.68814585101307729209096919475996e-02 5.34198211533226330338663956354139e-01 1.62924472819564610491516187806837e-03
            5.34198211533226330338663956354139e-01 4.48920329956642827351487312625977e-01 1.62924472819564610491516187806837e-03
            5.34198211533226330338663956354139e-01 1.68814585101307729209096919475996e-02 1.62924472819564610491516187806837e-03
            3.97083379134988478931234112678794e-01 4.14766785336368876135892946876993e-02 2.39061749419019640636174806047620e-03
            3.97083379134988478931234112678794e-01 5.61439942331374619577388784819050e-01 2.39061749419019640636174806047620e-03
            4.14766785336368876135892946876993e-02 3.97083379134988478931234112678794e-01 2.39061749419019640636174806047620e-03
            4.14766785336368876135892946876993e-02 5.61439942331374619577388784819050e-01 2.39061749419019640636174806047620e-03
            5.61439942331374619577388784819050e-01 3.97083379134988478931234112678794e-01 2.39061749419019640636174806047620e-03
            5.61439942331374619577388784819050e-01 4.14766785336368876135892946876993e-02 2.39061749419019640636174806047620e-03
            8.62263440628871780102571165116387e-02 1.37345861622121794692885643485170e-02 8.04385951443430027157921458069723e-04
            8.62263440628871780102571165116387e-02 9.00039069774900668541306458791951e-01 8.04385951443430027157921458069723e-04
            1.37345861622121794692885643485170e-02 8.62263440628871780102571165116387e-02 8.04385951443430027157921458069723e-04
            1.37345861622121794692885643485170e-02 9.00039069774900668541306458791951e-01 8.04385951443430027157921458069723e-04
            9.00039069774900668541306458791951e-01 8.62263440628871780102571165116387e-02 8.04385951443430027157921458069723e-04
            9.00039069774900668541306458791951e-01 1.37345861622121794692885643485170e-02 8.04385951443430027157921458069723e-04
            2.81487959942281984471179612228298e-01 1.79186043302918159014769372561204e-02 1.49128325264966544311251439580701e-03
            2.81487959942281984471179612228298e-01 7.00593435727426272485729441541480e-01 1.49128325264966544311251439580701e-03
            1.79186043302918159014769372561204e-02 2.81487959942281984471179612228298e-01 1.49128325264966544311251439580701e-03
            1.79186043302918159014769372561204e-02 7.00593435727426272485729441541480e-01 1.49128325264966544311251439580701e-03
            7.00593435727426272485729441541480e-01 2.81487959942281984471179612228298e-01 1.49128325264966544311251439580701e-03
            7.00593435727426272485729441541480e-01 1.79186043302918159014769372561204e-02 1.49128325264966544311251439580701e-03
            1.92376834744259289644574550948164e-01 1.28471390056792922340633822386735e-01 3.04697442177643109187390457748279e-03
            1.92376834744259289644574550948164e-01 6.79151775198947760259216011036187e-01 3.04697442177643109187390457748279e-03
            1.28471390056792922340633822386735e-01 1.92376834744259289644574550948164e-01 3.04697442177643109187390457748279e-03
            1.28471390056792922340633822386735e-01 6.79151775198947760259216011036187e-01 3.04697442177643109187390457748279e-03
            6.79151775198947760259216011036187e-01 1.92376834744259289644574550948164e-01 3.04697442177643109187390457748279e-03
            6.79151775198947760259216011036187e-01 1.28471390056792922340633822386735e-01 3.04697442177643109187390457748279e-03
            2.51794247517433122940389012001106e-01 1.37321159731595615616228656108433e-01 3.54367037302832183984224734274449e-03
            2.51794247517433122940389012001106e-01 6.10884592750971289198957947519375e-01 3.54367037302832183984224734274449e-03
            1.37321159731595615616228656108433e-01 2.51794247517433122940389012001106e-01 3.54367037302832183984224734274449e-03
            1.37321159731595615616228656108433e-01 6.10884592750971289198957947519375e-01 3.54367037302832183984224734274449e-03
            6.10884592750971289198957947519375e-01 2.51794247517433122940389012001106e-01 3.54367037302832183984224734274449e-03
            6.10884592750971289198957947519375e-01 1.37321159731595615616228656108433e-01 3.54367037302832183984224734274449e-03
            3.65034794936327955383603693917394e-01 1.74418264943838295655265113737187e-02 1.57487631404059475435930348652391e-03
            3.65034794936327955383603693917394e-01 6.17523378569288183825847227126360e-01 1.57487631404059475435930348652391e-03
            1.74418264943838295655265113737187e-02 3.65034794936327955383603693917394e-01 1.57487631404059475435930348652391e-03
            1.74418264943838295655265113737187e-02 6.17523378569288183825847227126360e-01 1.57487631404059475435930348652391e-03
            6.17523378569288183825847227126360e-01 3.65034794936327955383603693917394e-01 1.57487631404059475435930348652391e-03
            6.17523378569288183825847227126360e-01 1.74418264943838295655265113737187e-02 1.57487631404059475435930348652391e-03
            2.57156635550253476196758128935471e-01 8.61981702781003389368663647474023e-02 3.12593497656047953486835311309733e-03
            2.57156635550253476196758128935471e-01 6.56645194171646240377526737574954e-01 3.12593497656047953486835311309733e-03
            8.61981702781003389368663647474023e-02 2.57156635550253476196758128935471e-01 3.12593497656047953486835311309733e-03
            8.61981702781003389368663647474023e-02 6.56645194171646240377526737574954e-01 3.12593497656047953486835311309733e-03
            6.56645194171646240377526737574954e-01 2.57156635550253476196758128935471e-01 3.12593497656047953486835311309733e-03
            6.56645194171646240377526737574954e-01 8.61981702781003389368663647474023e-02 3.12593497656047953486835311309733e-03
            1.24181611934111021944282526874304e-01 3.41126780083469841389764987127364e-03 4.74652661273614420202326957820560e-04
            1.24181611934111021944282526874304e-01 8.72407120265054292218565024086274e-01 4.74652661273614420202326957820560e-04
            3.41126780083469841389764987127364e-03 1.24181611934111021944282526874304e-01 4.74652661273614420202326957820560e-04
            3.41126780083469841389764987127364e-03 8.72407120265054292218565024086274e-01 4.74652661273614420202326957820560e-04
            8.72407120265054292218565024086274e-01 1.24181611934111021944282526874304e-01 4.74652661273614420202326957820560e-04
            8.72407120265054292218565024086274e-01 3.41126780083469841389764987127364e-03 4.74652661273614420202326957820560e-04
            1.71650727809369157972341213280743e-01 4.45555167117895126183135801056778e-02 2.01562604214640178276751747432627e-03
            1.71650727809369157972341213280743e-01 7.83793755478841336348239110520808e-01 2.01562604214640178276751747432627e-03
            4.45555167117895126183135801056778e-02 1.71650727809369157972341213280743e-01 2.01562604214640178276751747432627e-03
            4.45555167117895126183135801056778e-02 7.83793755478841336348239110520808e-01 2.01562604214640178276751747432627e-03
            7.83793755478841336348239110520808e-01 1.71650727809369157972341213280743e-01 2.01562604214640178276751747432627e-03
            7.83793755478841336348239110520808e-01 4.45555167117895126183135801056778e-02 2.01562604214640178276751747432627e-03
            3.97723109198285573739894971367903e-01 1.22097548884562687177002260341396e-01 4.08610877411519878010803807910634e-03
            3.97723109198285573739894971367903e-01 4.80179341917151725205314960476244e-01 4.08610877411519878010803807910634e-03
            1.22097548884562687177002260341396e-01 3.97723109198285573739894971367903e-01 4.08610877411519878010803807910634e-03
            1.22097548884562687177002260341396e-01 4.80179341917151725205314960476244e-01 4.08610877411519878010803807910634e-03
            4.80179341917151725205314960476244e-01 3.97723109198285573739894971367903e-01 4.08610877411519878010803807910634e-03
            4.80179341917151725205314960476244e-01 1.22097548884562687177002260341396e-01 4.08610877411519878010803807910634e-03
            2.57896645669856483529258639464388e-01 1.95710371236428148922215086713550e-01 4.47932951436091821711960037077915e-03
            2.57896645669856483529258639464388e-01 5.46392983093715312037375042564236e-01 4.47932951436091821711960037077915e-03
            1.95710371236428148922215086713550e-01 2.57896645669856483529258639464388e-01 4.47932951436091821711960037077915e-03
            1.95710371236428148922215086713550e-01 5.46392983093715312037375042564236e-01 4.47932951436091821711960037077915e-03
            5.46392983093715312037375042564236e-01 2.57896645669856483529258639464388e-01 4.47932951436091821711960037077915e-03
            5.46392983093715312037375042564236e-01 1.95710371236428148922215086713550e-01 4.47932951436091821711960037077915e-03
            1.44150074481193601894091216308880e-01 1.83113919798339880307125326908135e-02 1.23104937074616088699130500572210e-03
            1.44150074481193601894091216308880e-01 8.37538533538972340686257211928023e-01 1.23104937074616088699130500572210e-03
            1.83113919798339880307125326908135e-02 1.44150074481193601894091216308880e-01 1.23104937074616088699130500572210e-03
            1.83113919798339880307125326908135e-02 8.37538533538972340686257211928023e-01 1.23104937074616088699130500572210e-03
            8.37538533538972340686257211928023e-01 1.44150074481193601894091216308880e-01 1.23104937074616088699130500572210e-03
            8.37538533538972340686257211928023e-01 1.83113919798339880307125326908135e-02 1.23104937074616088699130500572210e-03
            4.12259561001791008383321468500071e-01 3.26225803520382127975718589141252e-03 7.10646460660964017670704340190468e-04
            4.12259561001791008383321468500071e-01 5.84478180963005189418879581353394e-01 7.10646460660964017670704340190468e-04
            3.26225803520382127975718589141252e-03 4.12259561001791008383321468500071e-01 7.10646460660964017670704340190468e-04
            3.26225803520382127975718589141252e-03 5.84478180963005189418879581353394e-01 7.10646460660964017670704340190468e-04
            5.84478180963005189418879581353394e-01 4.12259561001791008383321468500071e-01 7.10646460660964017670704340190468e-04
            5.84478180963005189418879581353394e-01 3.26225803520382127975718589141252e-03 7.10646460660964017670704340190468e-04
            2.52428464791436324432538640394341e-01 3.47544150731571212936676928961788e-03 6.52578247959340355596746796607022e-04
            2.52428464791436324432538640394341e-01 7.44096093701247962570732852327637e-01 6.52578247959340355596746796607022e-04
            3.47544150731571212936676928961788e-03 2.52428464791436324432538640394341e-01 6.52578247959340355596746796607022e-04
            3.47544150731571212936676928961788e-03 7.44096093701247962570732852327637e-01 6.52578247959340355596746796607022e-04
            7.44096093701247962570732852327637e-01 2.52428464791436324432538640394341e-01 6.52578247959340355596746796607022e-04
            7.44096093701247962570732852327637e-01 3.47544150731571212936676928961788e-03 6.52578247959340355596746796607022e-04
            3.29366927485774507733395921604824e-01 3.53879008258183198887270037857888e-03 7.22954763894363607729109943988988e-04
            3.29366927485774507733395921604824e-01 6.67094282431643703645818277436774e-01 7.22954763894363607729109943988988e-04
            3.53879008258183198887270037857888e-03 3.29366927485774507733395921604824e-01 7.22954763894363607729109943988988e-04
            3.53879008258183198887270037857888e-03 6.67094282431643703645818277436774e-01 7.22954763894363607729109943988988e-04
            6.67094282431643703645818277436774e-01 3.29366927485774507733395921604824e-01 7.22954763894363607729109943988988e-04
            6.67094282431643703645818277436774e-01 3.53879008258183198887270037857888e-03 7.22954763894363607729109943988988e-04
            2.09581481700515409594132165693736e-01 1.88449390557523001332729251089404e-02 1.46729752903976013583831061737328e-03
            2.09581481700515409594132165693736e-01 7.71573579243732265986466245522024e-01 1.46729752903976013583831061737328e-03
            1.88449390557523001332729251089404e-02 2.09581481700515409594132165693736e-01 1.46729752903976013583831061737328e-03
            1.88449390557523001332729251089404e-02 7.71573579243732265986466245522024e-01 1.46729752903976013583831061737328e-03
            7.71573579243732265986466245522024e-01 2.09581481700515409594132165693736e-01 1.46729752903976013583831061737328e-03
            7.71573579243732265986466245522024e-01 1.88449390557523001332729251089404e-02 1.46729752903976013583831061737328e-03
            1.83640169446232209837788218464993e-01 3.63810210869633648272936099488106e-03 5.98229502634915494968526061825287e-04
            1.83640169446232209837788218464993e-01 8.12721728445071511792718865763163e-01 5.98229502634915494968526061825287e-04
            3.63810210869633648272936099488106e-03 1.83640169446232209837788218464993e-01 5.98229502634915494968526061825287e-04
            3.63810210869633648272936099488106e-03 8.12721728445071511792718865763163e-01 5.98229502634915494968526061825287e-04
            8.12721728445071511792718865763163e-01 1.83640169446232209837788218464993e-01 5.98229502634915494968526061825287e-04
            8.12721728445071511792718865763163e-01 3.63810210869633648272936099488106e-03 5.98229502634915494968526061825287e-04
            2.41696232469337524140229334079777e-01 4.57532869613174186795845344022382e-02 2.43675220996146524310432823767769e-03
            2.41696232469337524140229334079777e-01 7.12550480569345112691337362775812e-01 2.43675220996146524310432823767769e-03
            4.57532869613174186795845344022382e-02 2.41696232469337524140229334079777e-01 2.43675220996146524310432823767769e-03
            4.57532869613174186795845344022382e-02 7.12550480569345112691337362775812e-01 2.43675220996146524310432823767769e-03
            7.12550480569345112691337362775812e-01 2.41696232469337524140229334079777e-01 2.43675220996146524310432823767769e-03
            7.12550480569345112691337362775812e-01 4.57532869613174186795845344022382e-02 2.43675220996146524310432823767769e-03
            ];
        
    case  38
        
        % ALG. DEG.:   38
        % PTS CARD.:  267
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.8e-15
        
        xyw=[
            4.59072136031764554964240687695565e-01 4.59072136031764554964240687695565e-01 1.58198222014196152387932592375819e-03
            4.59072136031764554964240687695565e-01 8.18557279364708900715186246088706e-02 1.58198222014196152387932592375819e-03
            8.18557279364708900715186246088706e-02 4.59072136031764554964240687695565e-01 1.58198222014196152387932592375819e-03
            3.93268006255360069722826210636413e-01 3.93268006255360069722826210636413e-01 2.93082910955432202407267006094571e-03
            3.93268006255360069722826210636413e-01 2.13463987489279860554347578727175e-01 2.93082910955432202407267006094571e-03
            2.13463987489279860554347578727175e-01 3.93268006255360069722826210636413e-01 2.93082910955432202407267006094571e-03
            4.68182589206935673153964216908207e-01 4.68182589206935673153964216908207e-01 1.96510845185057228187708311395454e-03
            4.68182589206935673153964216908207e-01 6.36348215861286536920715661835857e-02 1.96510845185057228187708311395454e-03
            6.36348215861286536920715661835857e-02 4.68182589206935673153964216908207e-01 1.96510845185057228187708311395454e-03
            1.75045040462672892023210380330056e-02 1.75045040462672892023210380330056e-02 3.49734883664800603325639727358976e-04
            1.75045040462672892023210380330056e-02 9.64990991907465400778676212212304e-01 3.49734883664800603325639727358976e-04
            9.64990991907465400778676212212304e-01 1.75045040462672892023210380330056e-02 3.49734883664800603325639727358976e-04
            1.45480114652860703650105733686360e-01 1.45480114652860703650105733686360e-01 2.60121125552193084573548986782043e-03
            1.45480114652860703650105733686360e-01 7.09039770694278592699788532627281e-01 2.60121125552193084573548986782043e-03
            7.09039770694278592699788532627281e-01 1.45480114652860703650105733686360e-01 2.60121125552193084573548986782043e-03
            3.04717426636049414090390285991816e-03 3.04717426636049414090390285991816e-03 6.20233313265518054670927261540214e-05
            3.04717426636049414090390285991816e-03 9.93905651467279049882108665769920e-01 6.20233313265518054670927261540214e-05
            9.93905651467279049882108665769920e-01 3.04717426636049414090390285991816e-03 6.20233313265518054670927261540214e-05
            5.87361601120219983807047015034186e-02 5.87361601120219983807047015034186e-02 1.25319110825724373478595730091456e-03
            5.87361601120219983807047015034186e-02 8.82527679775956030994166212622076e-01 1.25319110825724373478595730091456e-03
            8.82527679775956030994166212622076e-01 5.87361601120219983807047015034186e-02 1.25319110825724373478595730091456e-03
            3.73835575017464616376372532613459e-02 3.73835575017464616376372532613459e-02 8.53775101957864220036942271008229e-04
            3.73835575017464616376372532613459e-02 9.25232884996507021213574262219481e-01 8.53775101957864220036942271008229e-04
            9.25232884996507021213574262219481e-01 3.73835575017464616376372532613459e-02 8.53775101957864220036942271008229e-04
            2.85504778291264027512141865372541e-01 2.85504778291264027512141865372541e-01 4.49280475940249442862173268053994e-03
            2.85504778291264027512141865372541e-01 4.28990443417471944975716269254917e-01 4.49280475940249442862173268053994e-03
            4.28990443417471944975716269254917e-01 2.85504778291264027512141865372541e-01 4.49280475940249442862173268053994e-03
            4.20333501770014217324700211975141e-01 4.20333501770014217324700211975141e-01 3.96885213756022810910328857403329e-03
            4.20333501770014217324700211975141e-01 1.59332996459971565350599576049717e-01 3.96885213756022810910328857403329e-03
            1.59332996459971565350599576049717e-01 4.20333501770014217324700211975141e-01 3.96885213756022810910328857403329e-03
            2.07689421653862921202104985241021e-01 2.07689421653862921202104985241021e-01 3.87427659633373874803474734562769e-03
            2.07689421653862921202104985241021e-01 5.84621156692274102084638798260130e-01 3.87427659633373874803474734562769e-03
            5.84621156692274102084638798260130e-01 2.07689421653862921202104985241021e-01 3.87427659633373874803474734562769e-03
            4.92196272955310765162550978857325e-01 4.92196272955310765162550978857325e-01 1.51794877726721681894528348522044e-03
            4.92196272955310765162550978857325e-01 1.56074540893784696748980422853492e-02 1.51794877726721681894528348522044e-03
            1.56074540893784696748980422853492e-02 4.92196272955310765162550978857325e-01 1.51794877726721681894528348522044e-03
            3.56565593983026118962698092218488e-01 3.56565593983026118962698092218488e-01 4.98799680519128318945121236538398e-03
            3.56565593983026118962698092218488e-01 2.86868812033947762074603815563023e-01 4.98799680519128318945121236538398e-03
            2.86868812033947762074603815563023e-01 3.56565593983026118962698092218488e-01 4.98799680519128318945121236538398e-03
            2.83765514701740717562472582358168e-01 5.22971139020114439421860197398928e-02 1.24250261557132582988793867428967e-03
            2.83765514701740717562472582358168e-01 6.63937371396247866250917013530852e-01 1.24250261557132582988793867428967e-03
            5.22971139020114439421860197398928e-02 2.83765514701740717562472582358168e-01 1.24250261557132582988793867428967e-03
            5.22971139020114439421860197398928e-02 6.63937371396247866250917013530852e-01 1.24250261557132582988793867428967e-03
            6.63937371396247866250917013530852e-01 2.83765514701740717562472582358168e-01 1.24250261557132582988793867428967e-03
            6.63937371396247866250917013530852e-01 5.22971139020114439421860197398928e-02 1.24250261557132582988793867428967e-03
            1.28516564969693936060224359607673e-01 9.90797627018412813137970829302503e-02 1.84824458035341416921315538246517e-03
            1.28516564969693936060224359607673e-01 7.72403672328464740992615134018706e-01 1.84824458035341416921315538246517e-03
            9.90797627018412813137970829302503e-02 1.28516564969693936060224359607673e-01 1.84824458035341416921315538246517e-03
            9.90797627018412813137970829302503e-02 7.72403672328464740992615134018706e-01 1.84824458035341416921315538246517e-03
            7.72403672328464740992615134018706e-01 1.28516564969693936060224359607673e-01 1.84824458035341416921315538246517e-03
            7.72403672328464740992615134018706e-01 9.90797627018412813137970829302503e-02 1.84824458035341416921315538246517e-03
            4.13075041428614328498092334029934e-02 2.06440514930518719677321470840070e-03 1.73852199496849705705886490747503e-04
            4.13075041428614328498092334029934e-02 9.56628090707833389494396669761045e-01 1.73852199496849705705886490747503e-04
            2.06440514930518719677321470840070e-03 4.13075041428614328498092334029934e-02 1.73852199496849705705886490747503e-04
            2.06440514930518719677321470840070e-03 9.56628090707833389494396669761045e-01 1.73852199496849705705886490747503e-04
            9.56628090707833389494396669761045e-01 4.13075041428614328498092334029934e-02 1.73852199496849705705886490747503e-04
            9.56628090707833389494396669761045e-01 2.06440514930518719677321470840070e-03 1.73852199496849705705886490747503e-04
            8.92719800534144680703718677250436e-02 3.67301756493483427812307695603522e-02 1.03691343697038932704368452419885e-03
            8.92719800534144680703718677250436e-02 8.73997844297237147515033939271234e-01 1.03691343697038932704368452419885e-03
            3.67301756493483427812307695603522e-02 8.92719800534144680703718677250436e-02 1.03691343697038932704368452419885e-03
            3.67301756493483427812307695603522e-02 8.73997844297237147515033939271234e-01 1.03691343697038932704368452419885e-03
            8.73997844297237147515033939271234e-01 8.92719800534144680703718677250436e-02 1.03691343697038932704368452419885e-03
            8.73997844297237147515033939271234e-01 3.67301756493483427812307695603522e-02 1.03691343697038932704368452419885e-03
            3.37205292317987237460386040766025e-01 2.25945824487785718037713422745583e-01 3.68552054500562598776669531730477e-03
            3.37205292317987237460386040766025e-01 4.36848883194227044501900536488392e-01 3.68552054500562598776669531730477e-03
            2.25945824487785718037713422745583e-01 3.37205292317987237460386040766025e-01 3.68552054500562598776669531730477e-03
            2.25945824487785718037713422745583e-01 4.36848883194227044501900536488392e-01 3.68552054500562598776669531730477e-03
            4.36848883194227044501900536488392e-01 3.37205292317987237460386040766025e-01 3.68552054500562598776669531730477e-03
            4.36848883194227044501900536488392e-01 2.25945824487785718037713422745583e-01 3.68552054500562598776669531730477e-03
            2.73851374124178703617360497446498e-01 3.27011819900537709093768512502720e-02 1.56269905721992657249286295950697e-03
            2.73851374124178703617360497446498e-01 6.93447443885767511595474843488773e-01 1.56269905721992657249286295950697e-03
            3.27011819900537709093768512502720e-02 2.73851374124178703617360497446498e-01 1.56269905721992657249286295950697e-03
            3.27011819900537709093768512502720e-02 6.93447443885767511595474843488773e-01 1.56269905721992657249286295950697e-03
            6.93447443885767511595474843488773e-01 2.73851374124178703617360497446498e-01 1.56269905721992657249286295950697e-03
            6.93447443885767511595474843488773e-01 3.27011819900537709093768512502720e-02 1.56269905721992657249286295950697e-03
            3.99162626294194411435967140278080e-02 1.30695090443300153404404539969619e-02 4.53088107226028440240039873998512e-04
            3.99162626294194411435967140278080e-02 9.47014228326250595557667111279443e-01 4.53088107226028440240039873998512e-04
            1.30695090443300153404404539969619e-02 3.99162626294194411435967140278080e-02 4.53088107226028440240039873998512e-04
            1.30695090443300153404404539969619e-02 9.47014228326250595557667111279443e-01 4.53088107226028440240039873998512e-04
            9.47014228326250595557667111279443e-01 3.99162626294194411435967140278080e-02 4.53088107226028440240039873998512e-04
            9.47014228326250595557667111279443e-01 1.30695090443300153404404539969619e-02 4.53088107226028440240039873998512e-04
            3.08106832759384330344687441538554e-01 7.37176145710159158586094463316840e-02 2.49781388487334667439232482877287e-03
            3.08106832759384330344687441538554e-01 6.18175552669599781552278727758676e-01 2.49781388487334667439232482877287e-03
            7.37176145710159158586094463316840e-02 3.08106832759384330344687441538554e-01 2.49781388487334667439232482877287e-03
            7.37176145710159158586094463316840e-02 6.18175552669599781552278727758676e-01 2.49781388487334667439232482877287e-03
            6.18175552669599781552278727758676e-01 3.08106832759384330344687441538554e-01 2.49781388487334667439232482877287e-03
            6.18175552669599781552278727758676e-01 7.37176145710159158586094463316840e-02 2.49781388487334667439232482877287e-03
            2.19624713138435623260846796256374e-01 2.70388221454352780867558259103589e-03 4.62834099424956190577118420037550e-04
            2.19624713138435623260846796256374e-01 7.77671404647020780842581189062912e-01 4.62834099424956190577118420037550e-04
            2.70388221454352780867558259103589e-03 2.19624713138435623260846796256374e-01 4.62834099424956190577118420037550e-04
            2.70388221454352780867558259103589e-03 7.77671404647020780842581189062912e-01 4.62834099424956190577118420037550e-04
            7.77671404647020780842581189062912e-01 2.19624713138435623260846796256374e-01 4.62834099424956190577118420037550e-04
            7.77671404647020780842581189062912e-01 2.70388221454352780867558259103589e-03 4.62834099424956190577118420037550e-04
            2.73464607114281443855929865094367e-01 2.19158263446028789278585691135959e-01 4.14108323134647609425895709023280e-03
            2.73464607114281443855929865094367e-01 5.07377129439689822376635675027501e-01 4.14108323134647609425895709023280e-03
            2.19158263446028789278585691135959e-01 2.73464607114281443855929865094367e-01 4.14108323134647609425895709023280e-03
            2.19158263446028789278585691135959e-01 5.07377129439689822376635675027501e-01 4.14108323134647609425895709023280e-03
            5.07377129439689822376635675027501e-01 2.73464607114281443855929865094367e-01 4.14108323134647609425895709023280e-03
            5.07377129439689822376635675027501e-01 2.19158263446028789278585691135959e-01 4.14108323134647609425895709023280e-03
            6.86830028555526772571226956642931e-02 2.07228118815140395503338055505083e-02 7.68421232903655489036853776241287e-04
            6.86830028555526772571226956642931e-02 9.10594185262933231150839219480986e-01 7.68421232903655489036853776241287e-04
            2.07228118815140395503338055505083e-02 6.86830028555526772571226956642931e-02 7.68421232903655489036853776241287e-04
            2.07228118815140395503338055505083e-02 9.10594185262933231150839219480986e-01 7.68421232903655489036853776241287e-04
            9.10594185262933231150839219480986e-01 6.86830028555526772571226956642931e-02 7.68421232903655489036853776241287e-04
            9.10594185262933231150839219480986e-01 2.07228118815140395503338055505083e-02 7.68421232903655489036853776241287e-04
            1.83013599840179452815647209718009e-01 1.00621720892076091380396007934905e-01 2.61137036940718670827155101221706e-03
            1.83013599840179452815647209718009e-01 7.16364679267744497437320205790456e-01 2.61137036940718670827155101221706e-03
            1.00621720892076091380396007934905e-01 1.83013599840179452815647209718009e-01 2.61137036940718670827155101221706e-03
            1.00621720892076091380396007934905e-01 7.16364679267744497437320205790456e-01 2.61137036940718670827155101221706e-03
            7.16364679267744497437320205790456e-01 1.83013599840179452815647209718009e-01 2.61137036940718670827155101221706e-03
            7.16364679267744497437320205790456e-01 1.00621720892076091380396007934905e-01 2.61137036940718670827155101221706e-03
            2.28799483229925271210447590419790e-01 6.94898178335092570057085481494141e-02 2.35919639420995507980705063744153e-03
            2.28799483229925271210447590419790e-01 7.01710698936565457906056053616339e-01 2.35919639420995507980705063744153e-03
            6.94898178335092570057085481494141e-02 2.28799483229925271210447590419790e-01 2.35919639420995507980705063744153e-03
            6.94898178335092570057085481494141e-02 7.01710698936565457906056053616339e-01 2.35919639420995507980705063744153e-03
            7.01710698936565457906056053616339e-01 2.28799483229925271210447590419790e-01 2.35919639420995507980705063744153e-03
            7.01710698936565457906056053616339e-01 6.94898178335092570057085481494141e-02 2.35919639420995507980705063744153e-03
            1.01882711486939128708861801442254e-01 6.83773554225138580076048810951761e-02 1.80105850247642843356410402577694e-03
            1.01882711486939128708861801442254e-01 8.29739933090546943894594278390286e-01 1.80105850247642843356410402577694e-03
            6.83773554225138580076048810951761e-02 1.01882711486939128708861801442254e-01 1.80105850247642843356410402577694e-03
            6.83773554225138580076048810951761e-02 8.29739933090546943894594278390286e-01 1.80105850247642843356410402577694e-03
            8.29739933090546943894594278390286e-01 1.01882711486939128708861801442254e-01 1.80105850247642843356410402577694e-03
            8.29739933090546943894594278390286e-01 6.83773554225138580076048810951761e-02 1.80105850247642843356410402577694e-03
            2.43563673395817054112555410938512e-01 1.43204413633164825275789766578782e-02 1.18514523369667602176180665196625e-03
            2.43563673395817054112555410938512e-01 7.42115885240866468564036040334031e-01 1.18514523369667602176180665196625e-03
            1.43204413633164825275789766578782e-02 2.43563673395817054112555410938512e-01 1.18514523369667602176180665196625e-03
            1.43204413633164825275789766578782e-02 7.42115885240866468564036040334031e-01 1.18514523369667602176180665196625e-03
            7.42115885240866468564036040334031e-01 2.43563673395817054112555410938512e-01 1.18514523369667602176180665196625e-03
            7.42115885240866468564036040334031e-01 1.43204413633164825275789766578782e-02 1.18514523369667602176180665196625e-03
            2.05887716517901192547768118856766e-01 3.89678567422942698228460756126879e-02 1.80524491668789866839761604211390e-03
            2.05887716517901192547768118856766e-01 7.55144426739804530690491901623318e-01 1.80524491668789866839761604211390e-03
            3.89678567422942698228460756126879e-02 2.05887716517901192547768118856766e-01 1.80524491668789866839761604211390e-03
            3.89678567422942698228460756126879e-02 7.55144426739804530690491901623318e-01 1.80524491668789866839761604211390e-03
            7.55144426739804530690491901623318e-01 2.05887716517901192547768118856766e-01 1.80524491668789866839761604211390e-03
            7.55144426739804530690491901623318e-01 3.89678567422942698228460756126879e-02 1.80524491668789866839761604211390e-03
            2.01869508887603077429417908206233e-01 1.49396461917944173292127629792958e-01 3.27022726537275166755480704239289e-03
            2.01869508887603077429417908206233e-01 6.48734029194452777034030077629723e-01 3.27022726537275166755480704239289e-03
            1.49396461917944173292127629792958e-01 2.01869508887603077429417908206233e-01 3.27022726537275166755480704239289e-03
            1.49396461917944173292127629792958e-01 6.48734029194452777034030077629723e-01 3.27022726537275166755480704239289e-03
            6.48734029194452777034030077629723e-01 2.01869508887603077429417908206233e-01 3.27022726537275166755480704239289e-03
            6.48734029194452777034030077629723e-01 1.49396461917944173292127629792958e-01 3.27022726537275166755480704239289e-03
            1.59116021010545172220673748597619e-01 6.22282118189532995611301657845615e-02 2.03961074015630156042178100506135e-03
            1.59116021010545172220673748597619e-01 7.78655767170501555973771701246733e-01 2.03961074015630156042178100506135e-03
            6.22282118189532995611301657845615e-02 1.59116021010545172220673748597619e-01 2.03961074015630156042178100506135e-03
            6.22282118189532995611301657845615e-02 7.78655767170501555973771701246733e-01 2.03961074015630156042178100506135e-03
            7.78655767170501555973771701246733e-01 1.59116021010545172220673748597619e-01 2.03961074015630156042178100506135e-03
            7.78655767170501555973771701246733e-01 6.22282118189532995611301657845615e-02 2.03961074015630156042178100506135e-03
            1.77947635855591812514830962754786e-01 1.76192414704859877383480437629260e-02 1.16281945905291768904310512056099e-03
            1.77947635855591812514830962754786e-01 8.04433122673922196277374041528674e-01 1.16281945905291768904310512056099e-03
            1.76192414704859877383480437629260e-02 1.77947635855591812514830962754786e-01 1.16281945905291768904310512056099e-03
            1.76192414704859877383480437629260e-02 8.04433122673922196277374041528674e-01 1.16281945905291768904310512056099e-03
            8.04433122673922196277374041528674e-01 1.77947635855591812514830962754786e-01 1.16281945905291768904310512056099e-03
            8.04433122673922196277374041528674e-01 1.76192414704859877383480437629260e-02 1.16281945905291768904310512056099e-03
            1.62036747949692083725636848612339e-01 3.94772766521245526555317084671515e-03 5.02872011041631604248702824833117e-04
            1.62036747949692083725636848612339e-01 8.34015524385095519122046425763983e-01 5.02872011041631604248702824833117e-04
            3.94772766521245526555317084671515e-03 1.62036747949692083725636848612339e-01 5.02872011041631604248702824833117e-04
            3.94772766521245526555317084671515e-03 8.34015524385095519122046425763983e-01 5.02872011041631604248702824833117e-04
            8.34015524385095519122046425763983e-01 1.62036747949692083725636848612339e-01 5.02872011041631604248702824833117e-04
            8.34015524385095519122046425763983e-01 3.94772766521245526555317084671515e-03 5.02872011041631604248702824833117e-04
            2.91854983858524597017947144195205e-01 2.93117988617291056266389581708154e-03 5.71580885705873253423148216967320e-04
            2.91854983858524597017947144195205e-01 7.05213836255302539690603680355707e-01 5.71580885705873253423148216967320e-04
            2.93117988617291056266389581708154e-03 2.91854983858524597017947144195205e-01 5.71580885705873253423148216967320e-04
            2.93117988617291056266389581708154e-03 7.05213836255302539690603680355707e-01 5.71580885705873253423148216967320e-04
            7.05213836255302539690603680355707e-01 2.91854983858524597017947144195205e-01 5.71580885705873253423148216967320e-04
            7.05213836255302539690603680355707e-01 2.93117988617291056266389581708154e-03 5.71580885705873253423148216967320e-04
            3.71878691749589151971377987138112e-01 2.77798098243856299988108204956916e-03 5.94230661254127579505579515029012e-04
            3.71878691749589151971377987138112e-01 6.25343327267972370897552991664270e-01 5.94230661254127579505579515029012e-04
            2.77798098243856299988108204956916e-03 3.71878691749589151971377987138112e-01 5.94230661254127579505579515029012e-04
            2.77798098243856299988108204956916e-03 6.25343327267972370897552991664270e-01 5.94230661254127579505579515029012e-04
            6.25343327267972370897552991664270e-01 3.71878691749589151971377987138112e-01 5.94230661254127579505579515029012e-04
            6.25343327267972370897552991664270e-01 2.77798098243856299988108204956916e-03 5.94230661254127579505579515029012e-04
            1.34969372652465280371103517609299e-01 3.43221858358502412977841800056922e-02 1.45260947260546073069031880464763e-03
            1.34969372652465280371103517609299e-01 8.30708441511684547720051341457292e-01 1.45260947260546073069031880464763e-03
            3.43221858358502412977841800056922e-02 1.34969372652465280371103517609299e-01 1.45260947260546073069031880464763e-03
            3.43221858358502412977841800056922e-02 8.30708441511684547720051341457292e-01 1.45260947260546073069031880464763e-03
            8.30708441511684547720051341457292e-01 1.34969372652465280371103517609299e-01 1.45260947260546073069031880464763e-03
            8.30708441511684547720051341457292e-01 3.43221858358502412977841800056922e-02 1.45260947260546073069031880464763e-03
            2.52453429906421944473038365686079e-01 1.08724039341924899604308052403212e-01 3.16335759879328654378682905701226e-03
            2.52453429906421944473038365686079e-01 6.38822530751653183678229197539622e-01 3.16335759879328654378682905701226e-03
            1.08724039341924899604308052403212e-01 2.52453429906421944473038365686079e-01 3.16335759879328654378682905701226e-03
            1.08724039341924899604308052403212e-01 6.38822530751653183678229197539622e-01 3.16335759879328654378682905701226e-03
            6.38822530751653183678229197539622e-01 2.52453429906421944473038365686079e-01 3.16335759879328654378682905701226e-03
            6.38822530751653183678229197539622e-01 1.08724039341924899604308052403212e-01 3.16335759879328654378682905701226e-03
            3.24235164108304496632939617484226e-01 1.49491079080454249833342927900048e-02 1.36284311036006036688128517653240e-03
            3.24235164108304496632939617484226e-01 6.60815727983650069710108709841734e-01 1.36284311036006036688128517653240e-03
            1.49491079080454249833342927900048e-02 3.24235164108304496632939617484226e-01 1.36284311036006036688128517653240e-03
            1.49491079080454249833342927900048e-02 6.60815727983650069710108709841734e-01 1.36284311036006036688128517653240e-03
            6.60815727983650069710108709841734e-01 3.24235164108304496632939617484226e-01 1.36284311036006036688128517653240e-03
            6.60815727983650069710108709841734e-01 1.49491079080454249833342927900048e-02 1.36284311036006036688128517653240e-03
            4.37416190646625679949721643424709e-01 3.73240771592634676401267768142134e-02 2.26079579990258157720650622479752e-03
            4.37416190646625679949721643424709e-01 5.25259732194110928737984522740589e-01 2.26079579990258157720650622479752e-03
            3.73240771592634676401267768142134e-02 4.37416190646625679949721643424709e-01 2.26079579990258157720650622479752e-03
            3.73240771592634676401267768142134e-02 5.25259732194110928737984522740589e-01 2.26079579990258157720650622479752e-03
            5.25259732194110928737984522740589e-01 4.37416190646625679949721643424709e-01 2.26079579990258157720650622479752e-03
            5.25259732194110928737984522740589e-01 3.73240771592634676401267768142134e-02 2.26079579990258157720650622479752e-03
            4.55998803661209317628077997142100e-01 2.96327193163695200642249183431431e-03 6.44587138714711118010758372776081e-04
            4.55998803661209317628077997142100e-01 5.41037924407153814065907226904528e-01 6.44587138714711118010758372776081e-04
            2.96327193163695200642249183431431e-03 4.55998803661209317628077997142100e-01 6.44587138714711118010758372776081e-04
            2.96327193163695200642249183431431e-03 5.41037924407153814065907226904528e-01 6.44587138714711118010758372776081e-04
            5.41037924407153814065907226904528e-01 4.55998803661209317628077997142100e-01 6.44587138714711118010758372776081e-04
            5.41037924407153814065907226904528e-01 2.96327193163695200642249183431431e-03 6.44587138714711118010758372776081e-04
            7.54872857005682990072514826351835e-02 4.20297816077350953195557181629738e-03 4.08341959086390329358023931760613e-04
            7.54872857005682990072514826351835e-02 9.20309736138658185389260779629694e-01 4.08341959086390329358023931760613e-04
            4.20297816077350953195557181629738e-03 7.54872857005682990072514826351835e-02 4.08341959086390329358023931760613e-04
            4.20297816077350953195557181629738e-03 9.20309736138658185389260779629694e-01 4.08341959086390329358023931760613e-04
            9.20309736138658185389260779629694e-01 7.54872857005682990072514826351835e-02 4.08341959086390329358023931760613e-04
            9.20309736138658185389260779629694e-01 4.20297816077350953195557181629738e-03 4.08341959086390329358023931760613e-04
            3.26054126652946585362968789922888e-01 1.14274187380454431139931159577827e-01 3.47388966177611087718379856426054e-03
            3.26054126652946585362968789922888e-01 5.59671685966598886352585395798087e-01 3.47388966177611087718379856426054e-03
            1.14274187380454431139931159577827e-01 3.26054126652946585362968789922888e-01 3.47388966177611087718379856426054e-03
            1.14274187380454431139931159577827e-01 5.59671685966598886352585395798087e-01 3.47388966177611087718379856426054e-03
            5.59671685966598886352585395798087e-01 3.26054126652946585362968789922888e-01 3.47388966177611087718379856426054e-03
            5.59671685966598886352585395798087e-01 1.14274187380454431139931159577827e-01 3.47388966177611087718379856426054e-03
            4.01199588505488202727633506583516e-01 1.12888085312921521152773607354902e-01 3.54930896974515889055745709868006e-03
            4.01199588505488202727633506583516e-01 4.85912326181590303875168501690496e-01 3.54930896974515889055745709868006e-03
            1.12888085312921521152773607354902e-01 4.01199588505488202727633506583516e-01 3.54930896974515889055745709868006e-03
            1.12888085312921521152773607354902e-01 4.85912326181590303875168501690496e-01 3.54930896974515889055745709868006e-03
            4.85912326181590303875168501690496e-01 4.01199588505488202727633506583516e-01 3.54930896974515889055745709868006e-03
            4.85912326181590303875168501690496e-01 1.12888085312921521152773607354902e-01 3.54930896974515889055745709868006e-03
            2.68760660341264001615968481928576e-01 1.60196677899755890450705919647589e-01 3.84596017428895809581179676683860e-03
            2.68760660341264001615968481928576e-01 5.71042661758980107933325598423835e-01 3.84596017428895809581179676683860e-03
            1.60196677899755890450705919647589e-01 2.68760660341264001615968481928576e-01 3.84596017428895809581179676683860e-03
            1.60196677899755890450705919647589e-01 5.71042661758980107933325598423835e-01 3.84596017428895809581179676683860e-03
            5.71042661758980107933325598423835e-01 2.68760660341264001615968481928576e-01 3.84596017428895809581179676683860e-03
            5.71042661758980107933325598423835e-01 1.60196677899755890450705919647589e-01 3.84596017428895809581179676683860e-03
            4.06014702573877173108485294505954e-01 1.51551286402894190563639398305895e-02 1.46260950936205245086207948901347e-03
            4.06014702573877173108485294505954e-01 5.78830168785833376610128198080929e-01 1.46260950936205245086207948901347e-03
            1.51551286402894190563639398305895e-02 4.06014702573877173108485294505954e-01 1.46260950936205245086207948901347e-03
            1.51551286402894190563639398305895e-02 5.78830168785833376610128198080929e-01 1.46260950936205245086207948901347e-03
            5.78830168785833376610128198080929e-01 4.06014702573877173108485294505954e-01 1.46260950936205245086207948901347e-03
            5.78830168785833376610128198080929e-01 1.51551286402894190563639398305895e-02 1.46260950936205245086207948901347e-03
            3.83510346834340154487108520697802e-01 7.00421815004378944813367979804752e-02 3.02834849824288633579616458746386e-03
            3.83510346834340154487108520697802e-01 5.46447471665221895520403450063895e-01 3.02834849824288633579616458746386e-03
            7.00421815004378944813367979804752e-02 3.83510346834340154487108520697802e-01 3.02834849824288633579616458746386e-03
            7.00421815004378944813367979804752e-02 5.46447471665221895520403450063895e-01 3.02834849824288633579616458746386e-03
            5.46447471665221895520403450063895e-01 3.83510346834340154487108520697802e-01 3.02834849824288633579616458746386e-03
            5.46447471665221895520403450063895e-01 7.00421815004378944813367979804752e-02 3.02834849824288633579616458746386e-03
            1.64470646369431552913376037849957e-02 3.44298850158080132349835977834118e-03 1.66852816559418681450696619883445e-04
            1.64470646369431552913376037849957e-02 9.80109946861476122315082193381386e-01 1.66852816559418681450696619883445e-04
            3.44298850158080132349835977834118e-03 1.64470646369431552913376037849957e-02 1.66852816559418681450696619883445e-04
            3.44298850158080132349835977834118e-03 9.80109946861476122315082193381386e-01 1.66852816559418681450696619883445e-04
            9.80109946861476122315082193381386e-01 1.64470646369431552913376037849957e-02 1.66852816559418681450696619883445e-04
            9.80109946861476122315082193381386e-01 3.44298850158080132349835977834118e-03 1.66852816559418681450696619883445e-04
            3.54297786162694750000667909262120e-01 3.76676144786773081651887196130701e-02 2.26473446047441642758890623099433e-03
            3.54297786162694750000667909262120e-01 6.08034599358627914078567755495897e-01 2.26473446047441642758890623099433e-03
            3.76676144786773081651887196130701e-02 3.54297786162694750000667909262120e-01 2.26473446047441642758890623099433e-03
            3.76676144786773081651887196130701e-02 6.08034599358627914078567755495897e-01 2.26473446047441642758890623099433e-03
            6.08034599358627914078567755495897e-01 3.54297786162694750000667909262120e-01 2.26473446047441642758890623099433e-03
            6.08034599358627914078567755495897e-01 3.76676144786773081651887196130701e-02 2.26473446047441642758890623099433e-03
            3.41995175798316008997090875709546e-01 1.65998426676715687966279233478417e-01 4.20126870715988726123768515208212e-03
            3.41995175798316008997090875709546e-01 4.92006397524968330792205506440951e-01 4.20126870715988726123768515208212e-03
            1.65998426676715687966279233478417e-01 3.41995175798316008997090875709546e-01 4.20126870715988726123768515208212e-03
            1.65998426676715687966279233478417e-01 4.92006397524968330792205506440951e-01 4.20126870715988726123768515208212e-03
            4.92006397524968330792205506440951e-01 3.41995175798316008997090875709546e-01 4.20126870715988726123768515208212e-03
            4.92006397524968330792205506440951e-01 1.65998426676715687966279233478417e-01 4.20126870715988726123768515208212e-03
            1.15288891079278379980799229542754e-01 1.20325311111651837847613677467962e-02 9.05485926201706186472462078995704e-04
            1.15288891079278379980799229542754e-01 8.72678577809556377253841219499009e-01 9.05485926201706186472462078995704e-04
            1.20325311111651837847613677467962e-02 1.15288891079278379980799229542754e-01 9.05485926201706186472462078995704e-04
            1.20325311111651837847613677467962e-02 8.72678577809556377253841219499009e-01 9.05485926201706186472462078995704e-04
            8.72678577809556377253841219499009e-01 1.15288891079278379980799229542754e-01 9.05485926201706186472462078995704e-04
            8.72678577809556377253841219499009e-01 1.20325311111651837847613677467962e-02 9.05485926201706186472462078995704e-04
            1.16528101812402543058233561623638e-01 2.98221681520129382886308277988974e-04 1.46142831591413282013580676732545e-04
            1.16528101812402543058233561623638e-01 8.83173676506077254266813270078273e-01 1.46142831591413282013580676732545e-04
            2.98221681520129382886308277988974e-04 1.16528101812402543058233561623638e-01 1.46142831591413282013580676732545e-04
            2.98221681520129382886308277988974e-04 8.83173676506077254266813270078273e-01 1.46142831591413282013580676732545e-04
            8.83173676506077254266813270078273e-01 1.16528101812402543058233561623638e-01 1.46142831591413282013580676732545e-04
            8.83173676506077254266813270078273e-01 2.98221681520129382886308277988974e-04 1.46142831591413282013580676732545e-04
            ];
        
    case  39
        
        % ALG. DEG.:   39
        % PTS CARD.:  282
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.6e-15
        
        xyw=[
            4.72409482150556669921570573933423e-01 4.72409482150556669921570573933423e-01 1.90023648880032858081434721242431e-03
            4.72409482150556669921570573933423e-01 5.51810356988866601568588521331549e-02 1.90023648880032858081434721242431e-03
            5.51810356988866601568588521331549e-02 4.72409482150556669921570573933423e-01 1.90023648880032858081434721242431e-03
            4.57507968673824261252036649239017e-01 4.57507968673824261252036649239017e-01 2.40451787008430202877140580142168e-03
            4.57507968673824261252036649239017e-01 8.49840626523514774959267015219666e-02 2.40451787008430202877140580142168e-03
            8.49840626523514774959267015219666e-02 4.57507968673824261252036649239017e-01 2.40451787008430202877140580142168e-03
            3.53847291916883299744966961952741e-01 3.53847291916883299744966961952741e-01 4.03949727267366963390227141417199e-03
            3.53847291916883299744966961952741e-01 2.92305416166233400510066076094517e-01 4.03949727267366963390227141417199e-03
            2.92305416166233400510066076094517e-01 3.53847291916883299744966961952741e-01 4.03949727267366963390227141417199e-03
            4.99106149860298997644747487356653e-01 4.99106149860298997644747487356653e-01 4.19238340914585027631444891227375e-04
            4.99106149860298997644747487356653e-01 1.78770027940200471050502528669313e-03 4.19238340914585027631444891227375e-04
            1.78770027940200471050502528669313e-03 4.99106149860298997644747487356653e-01 4.19238340914585027631444891227375e-04
            1.13331658196365314683617953050998e-01 1.13331658196365314683617953050998e-01 1.85446781068809006011444129313759e-03
            1.13331658196365314683617953050998e-01 7.73336683607269370632764093898004e-01 1.85446781068809006011444129313759e-03
            7.73336683607269370632764093898004e-01 1.13331658196365314683617953050998e-01 1.85446781068809006011444129313759e-03
            1.57787047228343108029591235208500e-02 1.57787047228343108029591235208500e-02 3.34303455844673681491424632739040e-04
            1.57787047228343108029591235208500e-02 9.68442590554331350638506137329387e-01 3.34303455844673681491424632739040e-04
            9.68442590554331350638506137329387e-01 1.57787047228343108029591235208500e-02 3.34303455844673681491424632739040e-04
            1.64009184766605586869303579078405e-01 1.64009184766605586869303579078405e-01 2.90176895980177960293611150177640e-03
            1.64009184766605586869303579078405e-01 6.71981630466788826261392841843190e-01 2.90176895980177960293611150177640e-03
            6.71981630466788826261392841843190e-01 1.64009184766605586869303579078405e-01 2.90176895980177960293611150177640e-03
            4.08471560729133986722416693737614e-01 4.08471560729133986722416693737614e-01 3.84960331613813482032337986993298e-03
            4.08471560729133986722416693737614e-01 1.83056878541732026555166612524772e-01 3.84960331613813482032337986993298e-03
            1.83056878541732026555166612524772e-01 4.08471560729133986722416693737614e-01 3.84960331613813482032337986993298e-03
            4.36541655107956616532760563131887e-01 4.36541655107956616532760563131887e-01 3.38825160904275340598146293302761e-03
            4.36541655107956616532760563131887e-01 1.26916689784086766934478873736225e-01 3.38825160904275340598146293302761e-03
            1.26916689784086766934478873736225e-01 4.36541655107956616532760563131887e-01 3.38825160904275340598146293302761e-03
            3.70326513393935824458580441387312e-02 3.70326513393935824458580441387312e-02 7.96024816034627885272256797577484e-04
            3.70326513393935824458580441387312e-02 9.25934697321212807352708296093624e-01 7.96024816034627885272256797577484e-04
            9.25934697321212807352708296093624e-01 3.70326513393935824458580441387312e-02 7.96024816034627885272256797577484e-04
            2.11731749565185534134315048504504e-01 2.11731749565185534134315048504504e-01 3.56278191707463378831710620886497e-03
            2.11731749565185534134315048504504e-01 5.76536500869628931731369902990991e-01 3.56278191707463378831710620886497e-03
            5.76536500869628931731369902990991e-01 2.11731749565185534134315048504504e-01 3.56278191707463378831710620886497e-03
            2.86957288459068593500234101156821e-01 2.86957288459068593500234101156821e-01 4.36436425490537883986608136410723e-03
            2.86957288459068593500234101156821e-01 4.26085423081862812999531797686359e-01 4.36436425490537883986608136410723e-03
            4.26085423081862812999531797686359e-01 2.86957288459068593500234101156821e-01 4.36436425490537883986608136410723e-03
            8.06765828829424935797476337029366e-02 8.06765828829424935797476337029366e-02 1.85384268403265112766520328335673e-03
            8.06765828829424935797476337029366e-02 8.38646834234115012840504732594127e-01 1.85384268403265112766520328335673e-03
            8.38646834234115012840504732594127e-01 8.06765828829424935797476337029366e-02 1.85384268403265112766520328335673e-03
            3.64321992072988865701832494892187e-03 3.64321992072988865701832494892187e-03 8.74538907766669241887480534813903e-05
            3.64321992072988865701832494892187e-03 9.92713560158540175848429498728365e-01 8.74538907766669241887480534813903e-05
            9.92713560158540175848429498728365e-01 3.64321992072988865701832494892187e-03 8.74538907766669241887480534813903e-05
            2.38275208584557002255976954074868e-01 4.37448243826125699601492513579615e-03 3.91201516419781921037046590328146e-04
            2.38275208584557002255976954074868e-01 7.57350308977181807534861945896409e-01 3.91201516419781921037046590328146e-04
            4.37448243826125699601492513579615e-03 2.38275208584557002255976954074868e-01 3.91201516419781921037046590328146e-04
            4.37448243826125699601492513579615e-03 7.57350308977181807534861945896409e-01 3.91201516419781921037046590328146e-04
            7.57350308977181807534861945896409e-01 2.38275208584557002255976954074868e-01 3.91201516419781921037046590328146e-04
            7.57350308977181807534861945896409e-01 4.37448243826125699601492513579615e-03 3.91201516419781921037046590328146e-04
            5.02290201314918416386490207514726e-01 2.31437075889000587469723768663243e-01 2.71296293947622833495070260312332e-03
            5.02290201314918416386490207514726e-01 2.66272722796080996143786023822031e-01 2.71296293947622833495070260312332e-03
            2.31437075889000587469723768663243e-01 5.02290201314918416386490207514726e-01 2.71296293947622833495070260312332e-03
            2.31437075889000587469723768663243e-01 2.66272722796080996143786023822031e-01 2.71296293947622833495070260312332e-03
            2.66272722796080996143786023822031e-01 5.02290201314918416386490207514726e-01 2.71296293947622833495070260312332e-03
            2.66272722796080996143786023822031e-01 2.31437075889000587469723768663243e-01 2.71296293947622833495070260312332e-03
            3.50682734157158282339850075004506e-01 2.36099447436519921383890618926671e-01 3.21678249161045553064508517593367e-03
            3.50682734157158282339850075004506e-01 4.13217818406321768520683690439910e-01 3.21678249161045553064508517593367e-03
            2.36099447436519921383890618926671e-01 3.50682734157158282339850075004506e-01 3.21678249161045553064508517593367e-03
            2.36099447436519921383890618926671e-01 4.13217818406321768520683690439910e-01 3.21678249161045553064508517593367e-03
            4.13217818406321768520683690439910e-01 3.50682734157158282339850075004506e-01 3.21678249161045553064508517593367e-03
            4.13217818406321768520683690439910e-01 2.36099447436519921383890618926671e-01 3.21678249161045553064508517593367e-03
            2.77808917719284009795188694624812e-01 2.66233974789171091296369198175853e-03 3.82255318815834753670007684078769e-04
            2.77808917719284009795188694624812e-01 7.19528742532824283628656303335447e-01 3.82255318815834753670007684078769e-04
            2.66233974789171091296369198175853e-03 2.77808917719284009795188694624812e-01 3.82255318815834753670007684078769e-04
            2.66233974789171091296369198175853e-03 7.19528742532824283628656303335447e-01 3.82255318815834753670007684078769e-04
            7.19528742532824283628656303335447e-01 2.77808917719284009795188694624812e-01 3.82255318815834753670007684078769e-04
            7.19528742532824283628656303335447e-01 2.66233974789171091296369198175853e-03 3.82255318815834753670007684078769e-04
            3.21203482682616781040962905535707e-01 2.07439611934945505877792015780869e-01 3.12974434051319170407845504655597e-03
            3.21203482682616781040962905535707e-01 4.71356905382437685325669463054510e-01 3.12974434051319170407845504655597e-03
            2.07439611934945505877792015780869e-01 3.21203482682616781040962905535707e-01 3.12974434051319170407845504655597e-03
            2.07439611934945505877792015780869e-01 4.71356905382437685325669463054510e-01 3.12974434051319170407845504655597e-03
            4.71356905382437685325669463054510e-01 3.21203482682616781040962905535707e-01 3.12974434051319170407845504655597e-03
            4.71356905382437685325669463054510e-01 2.07439611934945505877792015780869e-01 3.12974434051319170407845504655597e-03
            4.57424216852982068459709807939362e-01 1.19091524767886494662327834248572e-02 1.08104370344727293562492320688762e-03
            4.57424216852982068459709807939362e-01 5.30666630670229233501800081285182e-01 1.08104370344727293562492320688762e-03
            1.19091524767886494662327834248572e-02 4.57424216852982068459709807939362e-01 1.08104370344727293562492320688762e-03
            1.19091524767886494662327834248572e-02 5.30666630670229233501800081285182e-01 1.08104370344727293562492320688762e-03
            5.30666630670229233501800081285182e-01 4.57424216852982068459709807939362e-01 1.08104370344727293562492320688762e-03
            5.30666630670229233501800081285182e-01 1.19091524767886494662327834248572e-02 1.08104370344727293562492320688762e-03
            1.91038342852184600539899861360027e-01 2.15696233889989567519829272157494e-03 3.45426989161139847382686607346614e-04
            1.91038342852184600539899861360027e-01 8.06804694808915479065092313248897e-01 3.45426989161139847382686607346614e-04
            2.15696233889989567519829272157494e-03 1.91038342852184600539899861360027e-01 3.45426989161139847382686607346614e-04
            2.15696233889989567519829272157494e-03 8.06804694808915479065092313248897e-01 3.45426989161139847382686607346614e-04
            8.06804694808915479065092313248897e-01 1.91038342852184600539899861360027e-01 3.45426989161139847382686607346614e-04
            8.06804694808915479065092313248897e-01 2.15696233889989567519829272157494e-03 3.45426989161139847382686607346614e-04
            2.06351940732836303116215503905551e-01 1.07257493751645838209185512823751e-01 2.19054804285410704284275951181371e-03
            2.06351940732836303116215503905551e-01 6.86390565515517914185750214528525e-01 2.19054804285410704284275951181371e-03
            1.07257493751645838209185512823751e-01 2.06351940732836303116215503905551e-01 2.19054804285410704284275951181371e-03
            1.07257493751645838209185512823751e-01 6.86390565515517914185750214528525e-01 2.19054804285410704284275951181371e-03
            6.86390565515517914185750214528525e-01 2.06351940732836303116215503905551e-01 2.19054804285410704284275951181371e-03
            6.86390565515517914185750214528525e-01 1.07257493751645838209185512823751e-01 2.19054804285410704284275951181371e-03
            2.61238382787471912216403779893881e-01 1.12438393291482544023018874668196e-01 2.44886953539576006366740301700702e-03
            2.61238382787471912216403779893881e-01 6.26323223921045557638365153252380e-01 2.44886953539576006366740301700702e-03
            1.12438393291482544023018874668196e-01 2.61238382787471912216403779893881e-01 2.44886953539576006366740301700702e-03
            1.12438393291482544023018874668196e-01 6.26323223921045557638365153252380e-01 2.44886953539576006366740301700702e-03
            6.26323223921045557638365153252380e-01 2.61238382787471912216403779893881e-01 2.44886953539576006366740301700702e-03
            6.26323223921045557638365153252380e-01 1.12438393291482544023018874668196e-01 2.44886953539576006366740301700702e-03
            1.59388899623120305903967164340429e-01 1.14090324996719685857193837819068e-01 2.21549482544702641337308612889956e-03
            1.59388899623120305903967164340429e-01 7.26520775380160022116626805654960e-01 2.21549482544702641337308612889956e-03
            1.14090324996719685857193837819068e-01 1.59388899623120305903967164340429e-01 2.21549482544702641337308612889956e-03
            1.14090324996719685857193837819068e-01 7.26520775380160022116626805654960e-01 2.21549482544702641337308612889956e-03
            7.26520775380160022116626805654960e-01 1.59388899623120305903967164340429e-01 2.21549482544702641337308612889956e-03
            7.26520775380160022116626805654960e-01 1.14090324996719685857193837819068e-01 2.21549482544702641337308612889956e-03
            3.98869513224066885026530826507951e-01 6.19448419160624624835165263903036e-02 2.34993921431397032387189405255867e-03
            3.98869513224066885026530826507951e-01 5.39185644859870638612164839287288e-01 2.34993921431397032387189405255867e-03
            6.19448419160624624835165263903036e-02 3.98869513224066885026530826507951e-01 2.34993921431397032387189405255867e-03
            6.19448419160624624835165263903036e-02 5.39185644859870638612164839287288e-01 2.34993921431397032387189405255867e-03
            5.39185644859870638612164839287288e-01 3.98869513224066885026530826507951e-01 2.34993921431397032387189405255867e-03
            5.39185644859870638612164839287288e-01 6.19448419160624624835165263903036e-02 2.34993921431397032387189405255867e-03
            3.12272579273360884055676933712675e-01 1.21962883225284618449180129573506e-01 2.84747357948343486622189146828532e-03
            3.12272579273360884055676933712675e-01 5.65764537501354469739567321084905e-01 2.84747357948343486622189146828532e-03
            1.21962883225284618449180129573506e-01 3.12272579273360884055676933712675e-01 2.84747357948343486622189146828532e-03
            1.21962883225284618449180129573506e-01 5.65764537501354469739567321084905e-01 2.84747357948343486622189146828532e-03
            5.65764537501354469739567321084905e-01 3.12272579273360884055676933712675e-01 2.84747357948343486622189146828532e-03
            5.65764537501354469739567321084905e-01 1.21962883225284618449180129573506e-01 2.84747357948343486622189146828532e-03
            3.64436262358476370515347753098467e-01 3.86447963475402486710486016363575e-02 1.86161499437441217176536500943484e-03
            3.64436262358476370515347753098467e-01 5.96918941293983373874709741357947e-01 1.86161499437441217176536500943484e-03
            3.86447963475402486710486016363575e-02 3.64436262358476370515347753098467e-01 1.86161499437441217176536500943484e-03
            3.86447963475402486710486016363575e-02 5.96918941293983373874709741357947e-01 1.86161499437441217176536500943484e-03
            5.96918941293983373874709741357947e-01 3.64436262358476370515347753098467e-01 1.86161499437441217176536500943484e-03
            5.96918941293983373874709741357947e-01 3.86447963475402486710486016363575e-02 1.86161499437441217176536500943484e-03
            2.30564834240443161661460180766881e-01 1.72339717685665617941115357325543e-02 1.13555331101312202354836067286215e-03
            2.30564834240443161661460180766881e-01 7.52201193990990280013875235454179e-01 1.13555331101312202354836067286215e-03
            1.72339717685665617941115357325543e-02 2.30564834240443161661460180766881e-01 1.13555331101312202354836067286215e-03
            1.72339717685665617941115357325543e-02 7.52201193990990280013875235454179e-01 1.13555331101312202354836067286215e-03
            7.52201193990990280013875235454179e-01 2.30564834240443161661460180766881e-01 1.13555331101312202354836067286215e-03
            7.52201193990990280013875235454179e-01 1.72339717685665617941115357325543e-02 1.13555331101312202354836067286215e-03
            3.92682833818939619474619462380360e-02 1.45623703030592060309045265853456e-02 4.75092827573918936780866806302015e-04
            3.92682833818939619474619462380360e-02 9.46169346315046899675849090272095e-01 4.75092827573918936780866806302015e-04
            1.45623703030592060309045265853456e-02 3.92682833818939619474619462380360e-02 4.75092827573918936780866806302015e-04
            1.45623703030592060309045265853456e-02 9.46169346315046899675849090272095e-01 4.75092827573918936780866806302015e-04
            9.46169346315046899675849090272095e-01 3.92682833818939619474619462380360e-02 4.75092827573918936780866806302015e-04
            9.46169346315046899675849090272095e-01 1.45623703030592060309045265853456e-02 4.75092827573918936780866806302015e-04
            3.58257733547722723166373270942131e-01 1.54161109997349499156626961848815e-01 3.71655031528527347167711880615570e-03
            3.58257733547722723166373270942131e-01 4.87581156454927777676999767209054e-01 3.71655031528527347167711880615570e-03
            1.54161109997349499156626961848815e-01 3.58257733547722723166373270942131e-01 3.71655031528527347167711880615570e-03
            1.54161109997349499156626961848815e-01 4.87581156454927777676999767209054e-01 3.71655031528527347167711880615570e-03
            4.87581156454927777676999767209054e-01 3.58257733547722723166373270942131e-01 3.71655031528527347167711880615570e-03
            4.87581156454927777676999767209054e-01 1.54161109997349499156626961848815e-01 3.71655031528527347167711880615570e-03
            3.80507882548312525994305133281159e-01 1.01140130678948808595762898221437e-01 3.09321526687093969684716476820086e-03
            3.80507882548312525994305133281159e-01 5.18351986772738637654356352868490e-01 3.09321526687093969684716476820086e-03
            1.01140130678948808595762898221437e-01 3.80507882548312525994305133281159e-01 3.09321526687093969684716476820086e-03
            1.01140130678948808595762898221437e-01 5.18351986772738637654356352868490e-01 3.09321526687093969684716476820086e-03
            5.18351986772738637654356352868490e-01 3.80507882548312525994305133281159e-01 3.09321526687093969684716476820086e-03
            5.18351986772738637654356352868490e-01 1.01140130678948808595762898221437e-01 3.09321526687093969684716476820086e-03
            1.31886575586608933985388603105093e-01 7.32858166481221984245308931349427e-02 1.91871114087192569733208991067386e-03
            1.31886575586608933985388603105093e-01 7.94827607765268839834504888131050e-01 1.91871114087192569733208991067386e-03
            7.32858166481221984245308931349427e-02 1.31886575586608933985388603105093e-01 1.91871114087192569733208991067386e-03
            7.32858166481221984245308931349427e-02 7.94827607765268839834504888131050e-01 1.91871114087192569733208991067386e-03
            7.94827607765268839834504888131050e-01 1.31886575586608933985388603105093e-01 1.91871114087192569733208991067386e-03
            7.94827607765268839834504888131050e-01 7.32858166481221984245308931349427e-02 1.91871114087192569733208991067386e-03
            1.72630255068886018232632295621443e-01 1.35174633972146174959449993480121e-02 9.47936482876536525184196779747481e-04
            1.72630255068886018232632295621443e-01 8.13852281533899324372782757563982e-01 9.47936482876536525184196779747481e-04
            1.35174633972146174959449993480121e-02 1.72630255068886018232632295621443e-01 9.47936482876536525184196779747481e-04
            1.35174633972146174959449993480121e-02 8.13852281533899324372782757563982e-01 9.47936482876536525184196779747481e-04
            8.13852281533899324372782757563982e-01 1.72630255068886018232632295621443e-01 9.47936482876536525184196779747481e-04
            8.13852281533899324372782757563982e-01 1.35174633972146174959449993480121e-02 9.47936482876536525184196779747481e-04
            7.07756358900085702501314699475188e-02 4.31486463812221776592537025862839e-02 1.15793858431617623962195384734741e-03
            7.07756358900085702501314699475188e-02 8.86075717728769252090614827466197e-01 1.15793858431617623962195384734741e-03
            4.31486463812221776592537025862839e-02 7.07756358900085702501314699475188e-02 1.15793858431617623962195384734741e-03
            4.31486463812221776592537025862839e-02 8.86075717728769252090614827466197e-01 1.15793858431617623962195384734741e-03
            8.86075717728769252090614827466197e-01 7.07756358900085702501314699475188e-02 1.15793858431617623962195384734741e-03
            8.86075717728769252090614827466197e-01 4.31486463812221776592537025862839e-02 1.15793858431617623962195384734741e-03
            4.73824731204699201025931643016520e-02 2.84833883835317481364457492531983e-03 2.44135381101898417703296995107110e-04
            4.73824731204699201025931643016520e-02 9.49769188041176826153844103828305e-01 2.44135381101898417703296995107110e-04
            2.84833883835317481364457492531983e-03 4.73824731204699201025931643016520e-02 2.44135381101898417703296995107110e-04
            2.84833883835317481364457492531983e-03 9.49769188041176826153844103828305e-01 2.44135381101898417703296995107110e-04
            9.49769188041176826153844103828305e-01 4.73824731204699201025931643016520e-02 2.44135381101898417703296995107110e-04
            9.49769188041176826153844103828305e-01 2.84833883835317481364457492531983e-03 2.44135381101898417703296995107110e-04
            2.97193248295744893994196900166571e-01 4.03974541969813680242218367766327e-02 1.96662350376744640353954274303305e-03
            2.97193248295744893994196900166571e-01 6.62409297507273731042687359149568e-01 1.96662350376744640353954274303305e-03
            4.03974541969813680242218367766327e-02 2.97193248295744893994196900166571e-01 1.96662350376744640353954274303305e-03
            4.03974541969813680242218367766327e-02 6.62409297507273731042687359149568e-01 1.96662350376744640353954274303305e-03
            6.62409297507273731042687359149568e-01 2.97193248295744893994196900166571e-01 1.96662350376744640353954274303305e-03
            6.62409297507273731042687359149568e-01 4.03974541969813680242218367766327e-02 1.96662350376744640353954274303305e-03
            4.43765239953598289446290436899289e-01 3.10025483435276659471835358772296e-02 1.86849159371713937315140352524168e-03
            4.43765239953598289446290436899289e-01 5.25232211702874041137079075269867e-01 1.86849159371713937315140352524168e-03
            3.10025483435276659471835358772296e-02 4.43765239953598289446290436899289e-01 1.86849159371713937315140352524168e-03
            3.10025483435276659471835358772296e-02 5.25232211702874041137079075269867e-01 1.86849159371713937315140352524168e-03
            5.25232211702874041137079075269867e-01 4.43765239953598289446290436899289e-01 1.86849159371713937315140352524168e-03
            5.25232211702874041137079075269867e-01 3.10025483435276659471835358772296e-02 1.86849159371713937315140352524168e-03
            2.29854890249822918502076163349557e-01 3.92854639000807109860957666569448e-02 1.76531343516777954050223087278937e-03
            2.29854890249822918502076163349557e-01 7.30859645850096328878464646550128e-01 1.76531343516777954050223087278937e-03
            3.92854639000807109860957666569448e-02 2.29854890249822918502076163349557e-01 1.76531343516777954050223087278937e-03
            3.92854639000807109860957666569448e-02 7.30859645850096328878464646550128e-01 1.76531343516777954050223087278937e-03
            7.30859645850096328878464646550128e-01 2.29854890249822918502076163349557e-01 1.76531343516777954050223087278937e-03
            7.30859645850096328878464646550128e-01 3.92854639000807109860957666569448e-02 1.76531343516777954050223087278937e-03
            2.51525427138907131663358995865565e-01 7.15641320496676891504961304235621e-02 2.43665074530694967733235323237295e-03
            2.51525427138907131663358995865565e-01 6.76910440811425151430569258081960e-01 2.43665074530694967733235323237295e-03
            7.15641320496676891504961304235621e-02 2.51525427138907131663358995865565e-01 2.43665074530694967733235323237295e-03
            7.15641320496676891504961304235621e-02 6.76910440811425151430569258081960e-01 2.43665074530694967733235323237295e-03
            6.76910440811425151430569258081960e-01 2.51525427138907131663358995865565e-01 2.43665074530694967733235323237295e-03
            6.76910440811425151430569258081960e-01 7.15641320496676891504961304235621e-02 2.43665074530694967733235323237295e-03
            2.20628791932752477134016544368933e-01 1.54394591780774664835362841586175e-01 3.03484621096303339854438618772292e-03
            2.20628791932752477134016544368933e-01 6.24976616286472830275044998415979e-01 3.03484621096303339854438618772292e-03
            1.54394591780774664835362841586175e-01 2.20628791932752477134016544368933e-01 3.03484621096303339854438618772292e-03
            1.54394591780774664835362841586175e-01 6.24976616286472830275044998415979e-01 3.03484621096303339854438618772292e-03
            6.24976616286472830275044998415979e-01 2.20628791932752477134016544368933e-01 3.03484621096303339854438618772292e-03
            6.24976616286472830275044998415979e-01 1.54394591780774664835362841586175e-01 3.03484621096303339854438618772292e-03
            4.16403182046817754713430304036592e-01 2.77347337277504936772776211739711e-03 5.50881790375419870195439209226151e-04
            4.16403182046817754713430304036592e-01 5.80823344580407296966484409495024e-01 5.50881790375419870195439209226151e-04
            2.77347337277504936772776211739711e-03 4.16403182046817754713430304036592e-01 5.50881790375419870195439209226151e-04
            2.77347337277504936772776211739711e-03 5.80823344580407296966484409495024e-01 5.50881790375419870195439209226151e-04
            5.80823344580407296966484409495024e-01 4.16403182046817754713430304036592e-01 5.50881790375419870195439209226151e-04
            5.80823344580407296966484409495024e-01 2.77347337277504936772776211739711e-03 5.50881790375419870195439209226151e-04
            2.78037243584027538023661918487051e-01 1.71845869035243042910465760542138e-01 3.65459140776114863383150499487328e-03
            2.78037243584027538023661918487051e-01 5.50116887380729502332599167857552e-01 3.65459140776114863383150499487328e-03
            1.71845869035243042910465760542138e-01 2.78037243584027538023661918487051e-01 3.65459140776114863383150499487328e-03
            1.71845869035243042910465760542138e-01 5.50116887380729502332599167857552e-01 3.65459140776114863383150499487328e-03
            5.50116887380729502332599167857552e-01 2.78037243584027538023661918487051e-01 3.65459140776114863383150499487328e-03
            5.50116887380729502332599167857552e-01 1.71845869035243042910465760542138e-01 3.65459140776114863383150499487328e-03
            3.42075832214508490647375538173947e-01 3.25241025804930876733078726203985e-03 5.86809763869502467280303559249433e-04
            3.42075832214508490647375538173947e-01 6.54671757527442110813353792764246e-01 5.86809763869502467280303559249433e-04
            3.25241025804930876733078726203985e-03 3.42075832214508490647375538173947e-01 5.86809763869502467280303559249433e-04
            3.25241025804930876733078726203985e-03 6.54671757527442110813353792764246e-01 5.86809763869502467280303559249433e-04
            6.54671757527442110813353792764246e-01 3.42075832214508490647375538173947e-01 5.86809763869502467280303559249433e-04
            6.54671757527442110813353792764246e-01 3.25241025804930876733078726203985e-03 5.86809763869502467280303559249433e-04
            3.76651522742024291723339501913870e-01 1.60811994772276506404651286175067e-02 1.41387321020510284275140122645098e-03
            3.76651522742024291723339501913870e-01 6.07267277780748071513983177283080e-01 1.41387321020510284275140122645098e-03
            1.60811994772276506404651286175067e-02 3.76651522742024291723339501913870e-01 1.41387321020510284275140122645098e-03
            1.60811994772276506404651286175067e-02 6.07267277780748071513983177283080e-01 1.41387321020510284275140122645098e-03
            6.07267277780748071513983177283080e-01 3.76651522742024291723339501913870e-01 1.41387321020510284275140122645098e-03
            6.07267277780748071513983177283080e-01 1.60811994772276506404651286175067e-02 1.41387321020510284275140122645098e-03
            1.12297916741235570547985389566747e-01 4.36591987048875543320214376308286e-02 1.40590977903145685629082706213921e-03
            1.12297916741235570547985389566747e-01 8.44042884553876882058887076709652e-01 1.40590977903145685629082706213921e-03
            4.36591987048875543320214376308286e-02 1.12297916741235570547985389566747e-01 1.40590977903145685629082706213921e-03
            4.36591987048875543320214376308286e-02 8.44042884553876882058887076709652e-01 1.40590977903145685629082706213921e-03
            8.44042884553876882058887076709652e-01 1.12297916741235570547985389566747e-01 1.40590977903145685629082706213921e-03
            8.44042884553876882058887076709652e-01 4.36591987048875543320214376308286e-02 1.40590977903145685629082706213921e-03
            1.86141155841773708878861270932248e-01 6.82956923610527899537459006751305e-02 2.18881088644219939237656191721726e-03
            1.86141155841773708878861270932248e-01 7.45563151797173473411817212763708e-01 2.18881088644219939237656191721726e-03
            6.82956923610527899537459006751305e-02 1.86141155841773708878861270932248e-01 2.18881088644219939237656191721726e-03
            6.82956923610527899537459006751305e-02 7.45563151797173473411817212763708e-01 2.18881088644219939237656191721726e-03
            7.45563151797173473411817212763708e-01 1.86141155841773708878861270932248e-01 2.18881088644219939237656191721726e-03
            7.45563151797173473411817212763708e-01 6.82956923610527899537459006751305e-02 2.18881088644219939237656191721726e-03
            3.21703398275187124610852151818108e-01 7.47225031108798287693417705668253e-02 2.88989258748446366242834670856610e-03
            3.21703398275187124610852151818108e-01 6.03574098613933074375381693243980e-01 2.88989258748446366242834670856610e-03
            7.47225031108798287693417705668253e-02 3.21703398275187124610852151818108e-01 2.88989258748446366242834670856610e-03
            7.47225031108798287693417705668253e-02 6.03574098613933074375381693243980e-01 2.88989258748446366242834670856610e-03
            6.03574098613933074375381693243980e-01 3.21703398275187124610852151818108e-01 2.88989258748446366242834670856610e-03
            6.03574098613933074375381693243980e-01 7.47225031108798287693417705668253e-02 2.88989258748446366242834670856610e-03
            1.66012685178866747914838697397499e-01 3.61553743643339420277271756276605e-02 1.66084415751654133897030618527424e-03
            1.66012685178866747914838697397499e-01 7.97831940456799282301858511345927e-01 1.66084415751654133897030618527424e-03
            3.61553743643339420277271756276605e-02 1.66012685178866747914838697397499e-01 1.66084415751654133897030618527424e-03
            3.61553743643339420277271756276605e-02 7.97831940456799282301858511345927e-01 1.66084415751654133897030618527424e-03
            7.97831940456799282301858511345927e-01 1.66012685178866747914838697397499e-01 1.66084415751654133897030618527424e-03
            7.97831940456799282301858511345927e-01 3.61553743643339420277271756276605e-02 1.66084415751654133897030618527424e-03
            1.93576952354284764767911042326887e-02 2.83309069924117659308726580036364e-03 1.61710719288366602244474967697840e-04
            1.93576952354284764767911042326887e-02 9.77809214065330278842225197877269e-01 1.61710719288366602244474967697840e-04
            2.83309069924117659308726580036364e-03 1.93576952354284764767911042326887e-02 1.61710719288366602244474967697840e-04
            2.83309069924117659308726580036364e-03 9.77809214065330278842225197877269e-01 1.61710719288366602244474967697840e-04
            9.77809214065330278842225197877269e-01 1.93576952354284764767911042326887e-02 1.61710719288366602244474967697840e-04
            9.77809214065330278842225197877269e-01 2.83309069924117659308726580036364e-03 1.61710719288366602244474967697840e-04
            7.19904092251254096312962360570964e-02 1.79498835307187742937884422644856e-02 7.79979255879456709506236666840095e-04
            7.19904092251254096312962360570964e-02 9.10059707244155746685976282606134e-01 7.79979255879456709506236666840095e-04
            1.79498835307187742937884422644856e-02 7.19904092251254096312962360570964e-02 7.79979255879456709506236666840095e-04
            1.79498835307187742937884422644856e-02 9.10059707244155746685976282606134e-01 7.79979255879456709506236666840095e-04
            9.10059707244155746685976282606134e-01 7.19904092251254096312962360570964e-02 7.79979255879456709506236666840095e-04
            9.10059707244155746685976282606134e-01 1.79498835307187742937884422644856e-02 7.79979255879456709506236666840095e-04
            1.33332716879897866801840677908331e-01 3.19253987057180636452424771221104e-03 4.34229828650841581140495950563718e-04
            1.33332716879897866801840677908331e-01 8.63474743249530329869401157338871e-01 4.34229828650841581140495950563718e-04
            3.19253987057180636452424771221104e-03 1.33332716879897866801840677908331e-01 4.34229828650841581140495950563718e-04
            3.19253987057180636452424771221104e-03 8.63474743249530329869401157338871e-01 4.34229828650841581140495950563718e-04
            8.63474743249530329869401157338871e-01 1.33332716879897866801840677908331e-01 4.34229828650841581140495950563718e-04
            8.63474743249530329869401157338871e-01 3.19253987057180636452424771221104e-03 4.34229828650841581140495950563718e-04
            2.99445529172993885058673413368524e-01 1.62924574190147324059818600971994e-02 1.37583912829826286727818995814232e-03
            2.99445529172993885058673413368524e-01 6.84262013407991420699261198024033e-01 1.37583912829826286727818995814232e-03
            1.62924574190147324059818600971994e-02 2.99445529172993885058673413368524e-01 1.37583912829826286727818995814232e-03
            1.62924574190147324059818600971994e-02 6.84262013407991420699261198024033e-01 1.37583912829826286727818995814232e-03
            6.84262013407991420699261198024033e-01 2.99445529172993885058673413368524e-01 1.37583912829826286727818995814232e-03
            6.84262013407991420699261198024033e-01 1.62924574190147324059818600971994e-02 1.37583912829826286727818995814232e-03
            1.17107885189153299032582822292170e-01 1.79153991813487192819032145507663e-02 1.02529312215662404654537009918158e-03
            1.17107885189153299032582822292170e-01 8.64976715629497960868832251435379e-01 1.02529312215662404654537009918158e-03
            1.79153991813487192819032145507663e-02 1.17107885189153299032582822292170e-01 1.02529312215662404654537009918158e-03
            1.79153991813487192819032145507663e-02 8.64976715629497960868832251435379e-01 1.02529312215662404654537009918158e-03
            8.64976715629497960868832251435379e-01 1.17107885189153299032582822292170e-01 1.02529312215662404654537009918158e-03
            8.64976715629497960868832251435379e-01 1.79153991813487192819032145507663e-02 1.02529312215662404654537009918158e-03
            8.53676016562654599306725344831648e-02 3.56724930413417637767303780549355e-03 3.92075062823016567015860234235447e-04
            8.53676016562654599306725344831648e-02 9.11065149039600341573930109007051e-01 3.92075062823016567015860234235447e-04
            3.56724930413417637767303780549355e-03 8.53676016562654599306725344831648e-02 3.92075062823016567015860234235447e-04
            3.56724930413417637767303780549355e-03 9.11065149039600341573930109007051e-01 3.92075062823016567015860234235447e-04
            9.11065149039600341573930109007051e-01 8.53676016562654599306725344831648e-02 3.92075062823016567015860234235447e-04
            9.11065149039600341573930109007051e-01 3.56724930413417637767303780549355e-03 3.92075062823016567015860234235447e-04
            ];
        
    case  40
        
        % ALG. DEG.:   40
        % PTS CARD.:  295
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.4e-15
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 4.39274222643156082857007760367196e-03
            1.57037453637930216054918730606005e-02 1.57037453637930216054918730606005e-02 3.12803309837426067596677370374891e-04
            1.57037453637930216054918730606005e-02 9.68592509272413915155652830435429e-01 3.12803309837426067596677370374891e-04
            9.68592509272413915155652830435429e-01 1.57037453637930216054918730606005e-02 3.12803309837426067596677370374891e-04
            2.93889962897443068889913320163032e-01 2.93889962897443068889913320163032e-01 4.29726447949341054666216876967155e-03
            2.93889962897443068889913320163032e-01 4.12220074205113862220173359673936e-01 4.29726447949341054666216876967155e-03
            4.12220074205113862220173359673936e-01 2.93889962897443068889913320163032e-01 4.29726447949341054666216876967155e-03
            3.70102868423741626635603552131215e-01 3.70102868423741626635603552131215e-01 4.29966617078100712262545002317893e-03
            3.70102868423741626635603552131215e-01 2.59794263152516746728792895737570e-01 4.29966617078100712262545002317893e-03
            2.59794263152516746728792895737570e-01 3.70102868423741626635603552131215e-01 4.29966617078100712262545002317893e-03
            2.52696314852206671552892203180818e-01 2.52696314852206671552892203180818e-01 4.01956779966072645648988981292860e-03
            2.52696314852206671552892203180818e-01 4.94607370295586656894215593638364e-01 4.01956779966072645648988981292860e-03
            4.94607370295586656894215593638364e-01 2.52696314852206671552892203180818e-01 4.01956779966072645648988981292860e-03
            2.10577129514804201049216203500691e-01 2.10577129514804201049216203500691e-01 3.60591179745443999454801442539065e-03
            2.10577129514804201049216203500691e-01 5.78845740970391542390416361740790e-01 3.60591179745443999454801442539065e-03
            5.78845740970391542390416361740790e-01 2.10577129514804201049216203500691e-01 3.60591179745443999454801442539065e-03
            3.05751062854184036537974478164870e-03 3.05751062854184036537974478164870e-03 6.14819425666628664077853483327374e-05
            3.05751062854184036537974478164870e-03 9.93884978742916369576221313764108e-01 6.14819425666628664077853483327374e-05
            9.93884978742916369576221313764108e-01 3.05751062854184036537974478164870e-03 6.14819425666628664077853483327374e-05
            4.03292222274262812664602506629308e-01 4.03292222274262812664602506629308e-01 4.03340248372595876202861830961410e-03
            4.03292222274262812664602506629308e-01 1.93415555451474374670794986741384e-01 4.03340248372595876202861830961410e-03
            1.93415555451474374670794986741384e-01 4.03292222274262812664602506629308e-01 4.03340248372595876202861830961410e-03
            4.76751646800292860906012037958135e-01 4.76751646800292860906012037958135e-01 2.33948811422646056612517639905491e-03
            4.76751646800292860906012037958135e-01 4.64967063994142781879759240837302e-02 2.33948811422646056612517639905491e-03
            4.64967063994142781879759240837302e-02 4.76751646800292860906012037958135e-01 2.33948811422646056612517639905491e-03
            6.48562451597025496630166685463337e-02 6.48562451597025496630166685463337e-02 1.35864928596238915270433267323824e-03
            6.48562451597025496630166685463337e-02 8.70287509680594872918391047278419e-01 1.35864928596238915270433267323824e-03
            8.70287509680594872918391047278419e-01 6.48562451597025496630166685463337e-02 1.35864928596238915270433267323824e-03
            1.68896379782961736371049710214720e-01 1.68896379782961736371049710214720e-01 3.11997101526517630482038256900523e-03
            1.68896379782961736371049710214720e-01 6.62207240434076527257900579570560e-01 3.11997101526517630482038256900523e-03
            6.62207240434076527257900579570560e-01 1.68896379782961736371049710214720e-01 3.11997101526517630482038256900523e-03
            9.62980341591576360915638588267029e-02 9.62980341591576360915638588267029e-02 1.94545468980248065053728634410390e-03
            9.62980341591576360915638588267029e-02 8.07403931681684783328023513604421e-01 1.94545468980248065053728634410390e-03
            8.07403931681684783328023513604421e-01 9.62980341591576360915638588267029e-02 1.94545468980248065053728634410390e-03
            4.90701569130203962831870967420400e-01 4.90701569130203962831870967420400e-01 1.51448455313863469123836225094237e-03
            4.90701569130203962831870967420400e-01 1.85968617395920743362580651591998e-02 1.51448455313863469123836225094237e-03
            1.85968617395920743362580651591998e-02 4.90701569130203962831870967420400e-01 1.51448455313863469123836225094237e-03
            4.98464579236597593592250632354990e-01 4.98464579236597593592250632354990e-01 6.02514031127760157739714497182604e-04
            4.98464579236597593592250632354990e-01 3.07084152680481281549873529002070e-03 6.02514031127760157739714497182604e-04
            3.07084152680481281549873529002070e-03 4.98464579236597593592250632354990e-01 6.02514031127760157739714497182604e-04
            4.32271977606806356853041961585404e-01 4.32271977606806356853041961585404e-01 3.61244286747945154980588000626085e-03
            4.32271977606806356853041961585404e-01 1.35456044786387286293916076829191e-01 3.61244286747945154980588000626085e-03
            1.35456044786387286293916076829191e-01 4.32271977606806356853041961585404e-01 3.61244286747945154980588000626085e-03
            1.30241606990209146665904427209171e-01 1.30241606990209146665904427209171e-01 2.56819586307022217422368903783081e-03
            1.30241606990209146665904427209171e-01 7.39516786019581706668191145581659e-01 2.56819586307022217422368903783081e-03
            7.39516786019581706668191145581659e-01 1.30241606990209146665904427209171e-01 2.56819586307022217422368903783081e-03
            4.56871737154322776763848423797754e-01 4.56871737154322776763848423797754e-01 3.04222693774763896026880338752107e-03
            4.56871737154322776763848423797754e-01 8.62565256913544464723031524044927e-02 3.04222693774763896026880338752107e-03
            8.62565256913544464723031524044927e-02 4.56871737154322776763848423797754e-01 3.04222693774763896026880338752107e-03
            1.67177838355710456363922844502667e-01 1.52794904901324161625164599342952e-03 1.88776183926785134925552256390802e-04
            1.67177838355710456363922844502667e-01 8.31294212595276360566742823721142e-01 1.88776183926785134925552256390802e-04
            1.52794904901324161625164599342952e-03 1.67177838355710456363922844502667e-01 1.88776183926785134925552256390802e-04
            1.52794904901324161625164599342952e-03 8.31294212595276360566742823721142e-01 1.88776183926785134925552256390802e-04
            8.31294212595276360566742823721142e-01 1.67177838355710456363922844502667e-01 1.88776183926785134925552256390802e-04
            8.31294212595276360566742823721142e-01 1.52794904901324161625164599342952e-03 1.88776183926785134925552256390802e-04
            3.71478099445237464060198817605851e-01 1.49718224721410841232538402323371e-02 7.75842484927518824533443364543928e-04
            3.71478099445237464060198817605851e-01 6.13550078082621408448460442741634e-01 7.75842484927518824533443364543928e-04
            1.49718224721410841232538402323371e-02 3.71478099445237464060198817605851e-01 7.75842484927518824533443364543928e-04
            1.49718224721410841232538402323371e-02 6.13550078082621408448460442741634e-01 7.75842484927518824533443364543928e-04
            6.13550078082621408448460442741634e-01 3.71478099445237464060198817605851e-01 7.75842484927518824533443364543928e-04
            6.13550078082621408448460442741634e-01 1.49718224721410841232538402323371e-02 7.75842484927518824533443364543928e-04
            1.59023884712373636540760912794212e-01 6.80425879912778096975101149723741e-03 4.32207280373297485914774185289389e-04
            1.59023884712373636540760912794212e-01 8.34171856488498564274891577952076e-01 4.32207280373297485914774185289389e-04
            6.80425879912778096975101149723741e-03 1.59023884712373636540760912794212e-01 4.32207280373297485914774185289389e-04
            6.80425879912778096975101149723741e-03 8.34171856488498564274891577952076e-01 4.32207280373297485914774185289389e-04
            8.34171856488498564274891577952076e-01 1.59023884712373636540760912794212e-01 4.32207280373297485914774185289389e-04
            8.34171856488498564274891577952076e-01 6.80425879912778096975101149723741e-03 4.32207280373297485914774185289389e-04
            4.62179329282665363187732054939261e-02 3.53139747015546109909500671619753e-02 4.98592465936584087506100448905499e-04
            4.62179329282665363187732054939261e-02 9.18468092370178790240231592179043e-01 4.98592465936584087506100448905499e-04
            3.53139747015546109909500671619753e-02 4.62179329282665363187732054939261e-02 4.98592465936584087506100448905499e-04
            3.53139747015546109909500671619753e-02 9.18468092370178790240231592179043e-01 4.98592465936584087506100448905499e-04
            9.18468092370178790240231592179043e-01 4.62179329282665363187732054939261e-02 4.98592465936584087506100448905499e-04
            9.18468092370178790240231592179043e-01 3.53139747015546109909500671619753e-02 4.98592465936584087506100448905499e-04
            1.14865167380080651637896949068818e-01 1.58398644760465636543989020879053e-03 2.04538515649023709729384390421103e-04
            1.14865167380080651637896949068818e-01 8.83550846172314718884877038362902e-01 2.04538515649023709729384390421103e-04
            1.58398644760465636543989020879053e-03 1.14865167380080651637896949068818e-01 2.04538515649023709729384390421103e-04
            1.58398644760465636543989020879053e-03 8.83550846172314718884877038362902e-01 2.04538515649023709729384390421103e-04
            8.83550846172314718884877038362902e-01 1.14865167380080651637896949068818e-01 2.04538515649023709729384390421103e-04
            8.83550846172314718884877038362902e-01 1.58398644760465636543989020879053e-03 2.04538515649023709729384390421103e-04
            3.34246008184714449296137672718032e-01 2.15166720844818110114360365514585e-02 1.02183358004221650112974817403710e-03
            3.34246008184714449296137672718032e-01 6.44237319730803670303487251658225e-01 1.02183358004221650112974817403710e-03
            2.15166720844818110114360365514585e-02 3.34246008184714449296137672718032e-01 1.02183358004221650112974817403710e-03
            2.15166720844818110114360365514585e-02 6.44237319730803670303487251658225e-01 1.02183358004221650112974817403710e-03
            6.44237319730803670303487251658225e-01 3.34246008184714449296137672718032e-01 1.02183358004221650112974817403710e-03
            6.44237319730803670303487251658225e-01 2.15166720844818110114360365514585e-02 1.02183358004221650112974817403710e-03
            7.23711649483900210100273397983983e-02 2.02526169256312012223464336102552e-03 2.06322166313811512349668531030034e-04
            7.23711649483900210100273397983983e-02 9.25603573359046860602461492817383e-01 2.06322166313811512349668531030034e-04
            2.02526169256312012223464336102552e-03 7.23711649483900210100273397983983e-02 2.06322166313811512349668531030034e-04
            2.02526169256312012223464336102552e-03 9.25603573359046860602461492817383e-01 2.06322166313811512349668531030034e-04
            9.25603573359046860602461492817383e-01 7.23711649483900210100273397983983e-02 2.06322166313811512349668531030034e-04
            9.25603573359046860602461492817383e-01 2.02526169256312012223464336102552e-03 2.06322166313811512349668531030034e-04
            2.23684246664788699554549111780943e-01 1.53612037490868675454480651865197e-02 9.57348377198418017640368926635119e-04
            2.23684246664788699554549111780943e-01 7.60954549586124406879150683380431e-01 9.57348377198418017640368926635119e-04
            1.53612037490868675454480651865197e-02 2.23684246664788699554549111780943e-01 9.57348377198418017640368926635119e-04
            1.53612037490868675454480651865197e-02 7.60954549586124406879150683380431e-01 9.57348377198418017640368926635119e-04
            7.60954549586124406879150683380431e-01 2.23684246664788699554549111780943e-01 9.57348377198418017640368926635119e-04
            7.60954549586124406879150683380431e-01 1.53612037490868675454480651865197e-02 9.57348377198418017640368926635119e-04
            1.11129016990584944535314093627676e-01 1.01659439479149828200466032512850e-02 5.82001567994971762132205839179733e-04
            1.11129016990584944535314093627676e-01 8.78705039061499992847359408187913e-01 5.82001567994971762132205839179733e-04
            1.01659439479149828200466032512850e-02 1.11129016990584944535314093627676e-01 5.82001567994971762132205839179733e-04
            1.01659439479149828200466032512850e-02 8.78705039061499992847359408187913e-01 5.82001567994971762132205839179733e-04
            8.78705039061499992847359408187913e-01 1.11129016990584944535314093627676e-01 5.82001567994971762132205839179733e-04
            8.78705039061499992847359408187913e-01 1.01659439479149828200466032512850e-02 5.82001567994971762132205839179733e-04
            2.76444541807790877818717945046956e-01 3.00138537838292723813893303486111e-02 1.45933685532618887849276578805302e-03
            2.76444541807790877818717945046956e-01 6.93541604408379863677680532418890e-01 1.45933685532618887849276578805302e-03
            3.00138537838292723813893303486111e-02 2.76444541807790877818717945046956e-01 1.45933685532618887849276578805302e-03
            3.00138537838292723813893303486111e-02 6.93541604408379863677680532418890e-01 1.45933685532618887849276578805302e-03
            6.93541604408379863677680532418890e-01 2.76444541807790877818717945046956e-01 1.45933685532618887849276578805302e-03
            6.93541604408379863677680532418890e-01 3.00138537838292723813893303486111e-02 1.45933685532618887849276578805302e-03
            1.69263355465675779942813505840604e-01 1.89700426400215629851864207466861e-02 9.71169086161179236603435693808706e-04
            1.69263355465675779942813505840604e-01 8.11766601894302719522045208577765e-01 9.71169086161179236603435693808706e-04
            1.89700426400215629851864207466861e-02 1.69263355465675779942813505840604e-01 9.71169086161179236603435693808706e-04
            1.89700426400215629851864207466861e-02 8.11766601894302719522045208577765e-01 9.71169086161179236603435693808706e-04
            8.11766601894302719522045208577765e-01 1.69263355465675779942813505840604e-01 9.71169086161179236603435693808706e-04
            8.11766601894302719522045208577765e-01 1.89700426400215629851864207466861e-02 9.71169086161179236603435693808706e-04
            1.19650955248758308901990687900252e-01 2.67407842221269662075222584007861e-02 1.02185803211649354690915281196339e-03
            1.19650955248758308901990687900252e-01 8.53608260529114648562654110719450e-01 1.02185803211649354690915281196339e-03
            2.67407842221269662075222584007861e-02 1.19650955248758308901990687900252e-01 1.02185803211649354690915281196339e-03
            2.67407842221269662075222584007861e-02 8.53608260529114648562654110719450e-01 1.02185803211649354690915281196339e-03
            8.53608260529114648562654110719450e-01 1.19650955248758308901990687900252e-01 1.02185803211649354690915281196339e-03
            8.53608260529114648562654110719450e-01 2.67407842221269662075222584007861e-02 1.02185803211649354690915281196339e-03
            2.83019097129268648593836132931756e-01 8.53818693537522336624301377838719e-04 2.28089429217444983463355945296769e-04
            2.83019097129268648593836132931756e-01 7.16127084177193840019981507794000e-01 2.28089429217444983463355945296769e-04
            8.53818693537522336624301377838719e-04 2.83019097129268648593836132931756e-01 2.28089429217444983463355945296769e-04
            8.53818693537522336624301377838719e-04 7.16127084177193840019981507794000e-01 2.28089429217444983463355945296769e-04
            7.16127084177193840019981507794000e-01 2.83019097129268648593836132931756e-01 2.28089429217444983463355945296769e-04
            7.16127084177193840019981507794000e-01 8.53818693537522336624301377838719e-04 2.28089429217444983463355945296769e-04
            7.04045501511467719524262065533549e-02 1.22584700439817353978533276404050e-02 5.59076325349262218816770442231245e-04
            7.04045501511467719524262065533549e-02 9.17336979804871499588614369713468e-01 5.59076325349262218816770442231245e-04
            1.22584700439817353978533276404050e-02 7.04045501511467719524262065533549e-02 5.59076325349262218816770442231245e-04
            1.22584700439817353978533276404050e-02 9.17336979804871499588614369713468e-01 5.59076325349262218816770442231245e-04
            9.17336979804871499588614369713468e-01 7.04045501511467719524262065533549e-02 5.59076325349262218816770442231245e-04
            9.17336979804871499588614369713468e-01 1.22584700439817353978533276404050e-02 5.59076325349262218816770442231245e-04
            2.16472098751412100892110856875661e-01 3.71700471134970150677290234852990e-02 1.51823620580618899589264714222736e-03
            2.16472098751412100892110856875661e-01 7.46357854135090814651221080566756e-01 1.51823620580618899589264714222736e-03
            3.71700471134970150677290234852990e-02 2.16472098751412100892110856875661e-01 1.51823620580618899589264714222736e-03
            3.71700471134970150677290234852990e-02 7.46357854135090814651221080566756e-01 1.51823620580618899589264714222736e-03
            7.46357854135090814651221080566756e-01 2.16472098751412100892110856875661e-01 1.51823620580618899589264714222736e-03
            7.46357854135090814651221080566756e-01 3.71700471134970150677290234852990e-02 1.51823620580618899589264714222736e-03
            7.69268815632590530206869061657926e-02 3.21942006634704513956535265606362e-02 9.70092839555703584895895996709214e-04
            7.69268815632590530206869061657926e-02 8.90878917773270440072508336015744e-01 9.70092839555703584895895996709214e-04
            3.21942006634704513956535265606362e-02 7.69268815632590530206869061657926e-02 9.70092839555703584895895996709214e-04
            3.21942006634704513956535265606362e-02 8.90878917773270440072508336015744e-01 9.70092839555703584895895996709214e-04
            8.90878917773270440072508336015744e-01 7.69268815632590530206869061657926e-02 9.70092839555703584895895996709214e-04
            8.90878917773270440072508336015744e-01 3.21942006634704513956535265606362e-02 9.70092839555703584895895996709214e-04
            4.29222732038551268995973941855482e-01 8.93215271131544107097166573794311e-03 9.57461742196478917538149389088176e-04
            4.29222732038551268995973941855482e-01 5.61845115250133253503861396893626e-01 9.57461742196478917538149389088176e-04
            8.93215271131544107097166573794311e-03 4.29222732038551268995973941855482e-01 9.57461742196478917538149389088176e-04
            8.93215271131544107097166573794311e-03 5.61845115250133253503861396893626e-01 9.57461742196478917538149389088176e-04
            5.61845115250133253503861396893626e-01 4.29222732038551268995973941855482e-01 9.57461742196478917538149389088176e-04
            5.61845115250133253503861396893626e-01 8.93215271131544107097166573794311e-03 9.57461742196478917538149389088176e-04
            3.25147924885204830935947484249482e-01 2.24045215227499733279259430673847e-01 4.11606702811379617651388329591100e-03
            3.25147924885204830935947484249482e-01 4.50806859887295408029217469447758e-01 4.11606702811379617651388329591100e-03
            2.24045215227499733279259430673847e-01 3.25147924885204830935947484249482e-01 4.11606702811379617651388329591100e-03
            2.24045215227499733279259430673847e-01 4.50806859887295408029217469447758e-01 4.11606702811379617651388329591100e-03
            4.50806859887295408029217469447758e-01 3.25147924885204830935947484249482e-01 4.11606702811379617651388329591100e-03
            4.50806859887295408029217469447758e-01 2.24045215227499733279259430673847e-01 4.11606702811379617651388329591100e-03
            2.78272710286936342249930476100417e-01 1.87626216990675664941790046214010e-01 3.76976024293510262805351196391257e-03
            2.78272710286936342249930476100417e-01 5.34101072722388048319430708943401e-01 3.76976024293510262805351196391257e-03
            1.87626216990675664941790046214010e-01 2.78272710286936342249930476100417e-01 3.76976024293510262805351196391257e-03
            1.87626216990675664941790046214010e-01 5.34101072722388048319430708943401e-01 3.76976024293510262805351196391257e-03
            5.34101072722388048319430708943401e-01 2.78272710286936342249930476100417e-01 3.76976024293510262805351196391257e-03
            5.34101072722388048319430708943401e-01 1.87626216990675664941790046214010e-01 3.76976024293510262805351196391257e-03
            2.85210253518710044051687191313249e-01 9.75852132386811378650381954003024e-03 9.27587312436341454927046790146505e-04
            2.85210253518710044051687191313249e-01 7.05031225157421892468789792474126e-01 9.27587312436341454927046790146505e-04
            9.75852132386811378650381954003024e-03 2.85210253518710044051687191313249e-01 9.27587312436341454927046790146505e-04
            9.75852132386811378650381954003024e-03 7.05031225157421892468789792474126e-01 9.27587312436341454927046790146505e-04
            7.05031225157421892468789792474126e-01 2.85210253518710044051687191313249e-01 9.27587312436341454927046790146505e-04
            7.05031225157421892468789792474126e-01 9.75852132386811378650381954003024e-03 9.27587312436341454927046790146505e-04
            2.72725631829831005603637095191516e-01 5.87062906216736027364966332697804e-02 2.20241068708403519779692736335619e-03
            2.72725631829831005603637095191516e-01 6.68568077548495343087608944188105e-01 2.20241068708403519779692736335619e-03
            5.87062906216736027364966332697804e-02 2.72725631829831005603637095191516e-01 2.20241068708403519779692736335619e-03
            5.87062906216736027364966332697804e-02 6.68568077548495343087608944188105e-01 2.20241068708403519779692736335619e-03
            6.68568077548495343087608944188105e-01 2.72725631829831005603637095191516e-01 2.20241068708403519779692736335619e-03
            6.68568077548495343087608944188105e-01 5.87062906216736027364966332697804e-02 2.20241068708403519779692736335619e-03
            4.15469594771239403296902992224204e-01 3.07419996861602630444654238317526e-02 1.90360250309295087750571795481846e-03
            4.15469594771239403296902992224204e-01 5.53788405542600226105776073382003e-01 1.90360250309295087750571795481846e-03
            3.07419996861602630444654238317526e-02 4.15469594771239403296902992224204e-01 1.90360250309295087750571795481846e-03
            3.07419996861602630444654238317526e-02 5.53788405542600226105776073382003e-01 1.90360250309295087750571795481846e-03
            5.53788405542600226105776073382003e-01 4.15469594771239403296902992224204e-01 1.90360250309295087750571795481846e-03
            5.53788405542600226105776073382003e-01 3.07419996861602630444654238317526e-02 1.90360250309295087750571795481846e-03
            2.20577446533709115028543124026328e-01 3.25675363711016695872357828989152e-03 4.84722289527270957989329991377758e-04
            2.20577446533709115028543124026328e-01 7.76165799829180658164773376483936e-01 4.84722289527270957989329991377758e-04
            3.25675363711016695872357828989152e-03 2.20577446533709115028543124026328e-01 4.84722289527270957989329991377758e-04
            3.25675363711016695872357828989152e-03 7.76165799829180658164773376483936e-01 4.84722289527270957989329991377758e-04
            7.76165799829180658164773376483936e-01 2.20577446533709115028543124026328e-01 4.84722289527270957989329991377758e-04
            7.76165799829180658164773376483936e-01 3.25675363711016695872357828989152e-03 4.84722289527270957989329991377758e-04
            3.42885620394358714424498657535878e-01 4.51876229604891294089341613471333e-02 2.11204016674039225104353434403492e-03
            3.42885620394358714424498657535878e-01 6.11926756645152170044354988931445e-01 2.11204016674039225104353434403492e-03
            4.51876229604891294089341613471333e-02 3.42885620394358714424498657535878e-01 2.11204016674039225104353434403492e-03
            4.51876229604891294089341613471333e-02 6.11926756645152170044354988931445e-01 2.11204016674039225104353434403492e-03
            6.11926756645152170044354988931445e-01 3.42885620394358714424498657535878e-01 2.11204016674039225104353434403492e-03
            6.11926756645152170044354988931445e-01 4.51876229604891294089341613471333e-02 2.11204016674039225104353434403492e-03
            2.31503860611680267744816319464007e-01 1.50818510224056762103472806302307e-01 3.30966277647021678981476000558359e-03
            2.31503860611680267744816319464007e-01 6.17677629164262942396135258604772e-01 3.30966277647021678981476000558359e-03
            1.50818510224056762103472806302307e-01 2.31503860611680267744816319464007e-01 3.30966277647021678981476000558359e-03
            1.50818510224056762103472806302307e-01 6.17677629164262942396135258604772e-01 3.30966277647021678981476000558359e-03
            6.17677629164262942396135258604772e-01 2.31503860611680267744816319464007e-01 3.30966277647021678981476000558359e-03
            6.17677629164262942396135258604772e-01 1.50818510224056762103472806302307e-01 3.30966277647021678981476000558359e-03
            3.81017129649182106909677258954616e-02 1.66736740374798728114846824155393e-02 5.00703644413419131191378408374248e-04
            3.81017129649182106909677258954616e-02 9.45224612997601965069804919039598e-01 5.00703644413419131191378408374248e-04
            1.66736740374798728114846824155393e-02 3.81017129649182106909677258954616e-02 5.00703644413419131191378408374248e-04
            1.66736740374798728114846824155393e-02 9.45224612997601965069804919039598e-01 5.00703644413419131191378408374248e-04
            9.45224612997601965069804919039598e-01 3.81017129649182106909677258954616e-02 5.00703644413419131191378408374248e-04
            9.45224612997601965069804919039598e-01 1.66736740374798728114846824155393e-02 5.00703644413419131191378408374248e-04
            3.52891108510167694767289958690526e-01 1.62105242145544004017665429273620e-01 3.77815553726075330220890435839465e-03
            3.52891108510167694767289958690526e-01 4.85003649344288301215044612035854e-01 3.77815553726075330220890435839465e-03
            1.62105242145544004017665429273620e-01 3.52891108510167694767289958690526e-01 3.77815553726075330220890435839465e-03
            1.62105242145544004017665429273620e-01 4.85003649344288301215044612035854e-01 3.77815553726075330220890435839465e-03
            4.85003649344288301215044612035854e-01 3.52891108510167694767289958690526e-01 3.77815553726075330220890435839465e-03
            4.85003649344288301215044612035854e-01 1.62105242145544004017665429273620e-01 3.77815553726075330220890435839465e-03
            4.00076069919581056577584377009771e-01 6.45258600999303599810019704818842e-02 2.64935252199681231174976936415533e-03
            4.00076069919581056577584377009771e-01 5.35398069980488555685838036879431e-01 2.64935252199681231174976936415533e-03
            6.45258600999303599810019704818842e-02 4.00076069919581056577584377009771e-01 2.64935252199681231174976936415533e-03
            6.45258600999303599810019704818842e-02 5.35398069980488555685838036879431e-01 2.64935252199681231174976936415533e-03
            5.35398069980488555685838036879431e-01 4.00076069919581056577584377009771e-01 2.64935252199681231174976936415533e-03
            5.35398069980488555685838036879431e-01 6.45258600999303599810019704818842e-02 2.64935252199681231174976936415533e-03
            1.87784341546537586475906778105127e-01 1.14379695468569117511314914281684e-01 2.77734489806421474059061615946575e-03
            1.87784341546537586475906778105127e-01 6.97835962984893365401717346685473e-01 2.77734489806421474059061615946575e-03
            1.14379695468569117511314914281684e-01 1.87784341546537586475906778105127e-01 2.77734489806421474059061615946575e-03
            1.14379695468569117511314914281684e-01 6.97835962984893365401717346685473e-01 2.77734489806421474059061615946575e-03
            6.97835962984893365401717346685473e-01 1.87784341546537586475906778105127e-01 2.77734489806421474059061615946575e-03
            6.97835962984893365401717346685473e-01 1.14379695468569117511314914281684e-01 2.77734489806421474059061615946575e-03
            2.07376937690341156539375333522912e-01 6.95719958527820786509110462247918e-02 2.25299228047472644559379162387813e-03
            2.07376937690341156539375333522912e-01 7.23051066456876778687501428066753e-01 2.25299228047472644559379162387813e-03
            6.95719958527820786509110462247918e-02 2.07376937690341156539375333522912e-01 2.25299228047472644559379162387813e-03
            6.95719958527820786509110462247918e-02 7.23051066456876778687501428066753e-01 2.25299228047472644559379162387813e-03
            7.23051066456876778687501428066753e-01 2.07376937690341156539375333522912e-01 2.25299228047472644559379162387813e-03
            7.23051066456876778687501428066753e-01 6.95719958527820786509110462247918e-02 2.25299228047472644559379162387813e-03
            1.07803830226375463308130520090344e-01 5.60343597005426088952795282693842e-02 1.55656756946014745990325423008471e-03
            1.07803830226375463308130520090344e-01 8.36161810073081879224332624289673e-01 1.55656756946014745990325423008471e-03
            5.60343597005426088952795282693842e-02 1.07803830226375463308130520090344e-01 1.55656756946014745990325423008471e-03
            5.60343597005426088952795282693842e-02 8.36161810073081879224332624289673e-01 1.55656756946014745990325423008471e-03
            8.36161810073081879224332624289673e-01 1.07803830226375463308130520090344e-01 1.55656756946014745990325423008471e-03
            8.36161810073081879224332624289673e-01 5.60343597005426088952795282693842e-02 1.55656756946014745990325423008471e-03
            1.47901466073126069966647833098250e-01 8.15896657841670980282344771694625e-02 2.19491910810134735476673029097583e-03
            1.47901466073126069966647833098250e-01 7.70508868142706804249542074103374e-01 2.19491910810134735476673029097583e-03
            8.15896657841670980282344771694625e-02 1.47901466073126069966647833098250e-01 2.19491910810134735476673029097583e-03
            8.15896657841670980282344771694625e-02 7.70508868142706804249542074103374e-01 2.19491910810134735476673029097583e-03
            7.70508868142706804249542074103374e-01 1.47901466073126069966647833098250e-01 2.19491910810134735476673029097583e-03
            7.70508868142706804249542074103374e-01 8.15896657841670980282344771694625e-02 2.19491910810134735476673029097583e-03
            1.60042668411358779412267949737725e-01 4.46292737712247408743415633125551e-02 1.61927618604146197206494761644535e-03
            1.60042668411358779412267949737725e-01 7.95328057817416556041223429929232e-01 1.61927618604146197206494761644535e-03
            4.46292737712247408743415633125551e-02 1.60042668411358779412267949737725e-01 1.61927618604146197206494761644535e-03
            4.46292737712247408743415633125551e-02 7.95328057817416556041223429929232e-01 1.61927618604146197206494761644535e-03
            7.95328057817416556041223429929232e-01 1.60042668411358779412267949737725e-01 1.61927618604146197206494761644535e-03
            7.95328057817416556041223429929232e-01 4.46292737712247408743415633125551e-02 1.61927618604146197206494761644535e-03
            3.77876669688772037769552980535082e-01 1.08791876599217379739670263916196e-01 3.28714905580476798516165182206805e-03
            3.77876669688772037769552980535082e-01 5.13331453712010499224049908661982e-01 3.28714905580476798516165182206805e-03
            1.08791876599217379739670263916196e-01 3.77876669688772037769552980535082e-01 3.28714905580476798516165182206805e-03
            1.08791876599217379739670263916196e-01 5.13331453712010499224049908661982e-01 3.28714905580476798516165182206805e-03
            5.13331453712010499224049908661982e-01 3.77876669688772037769552980535082e-01 3.28714905580476798516165182206805e-03
            5.13331453712010499224049908661982e-01 1.08791876599217379739670263916196e-01 3.28714905580476798516165182206805e-03
            3.92233696648084315117266385186667e-02 3.20614858452418710968445658693327e-03 2.30968790187327586808929691386538e-04
            3.92233696648084315117266385186667e-02 9.57570481750667390485887153772637e-01 2.30968790187327586808929691386538e-04
            3.20614858452418710968445658693327e-03 3.92233696648084315117266385186667e-02 2.30968790187327586808929691386538e-04
            3.20614858452418710968445658693327e-03 9.57570481750667390485887153772637e-01 2.30968790187327586808929691386538e-04
            9.57570481750667390485887153772637e-01 3.92233696648084315117266385186667e-02 2.30968790187327586808929691386538e-04
            9.57570481750667390485887153772637e-01 3.20614858452418710968445658693327e-03 2.30968790187327586808929691386538e-04
            3.24304445077138125963500669968198e-01 8.29765480589191445348618003663432e-02 2.81016493113481809901221808445371e-03
            3.24304445077138125963500669968198e-01 5.92719006863942632357122874964261e-01 2.81016493113481809901221808445371e-03
            8.29765480589191445348618003663432e-02 3.24304445077138125963500669968198e-01 2.81016493113481809901221808445371e-03
            8.29765480589191445348618003663432e-02 5.92719006863942632357122874964261e-01 2.81016493113481809901221808445371e-03
            5.92719006863942632357122874964261e-01 3.24304445077138125963500669968198e-01 2.81016493113481809901221808445371e-03
            5.92719006863942632357122874964261e-01 8.29765480589191445348618003663432e-02 2.81016493113481809901221808445371e-03
            3.50984847622504703146262272639433e-01 3.52073500998824817295740352562916e-03 6.16384778464870676170650476421997e-04
            3.50984847622504703146262272639433e-01 6.45494417367507011817906459327787e-01 6.16384778464870676170650476421997e-04
            3.52073500998824817295740352562916e-03 3.50984847622504703146262272639433e-01 6.16384778464870676170650476421997e-04
            3.52073500998824817295740352562916e-03 6.45494417367507011817906459327787e-01 6.16384778464870676170650476421997e-04
            6.45494417367507011817906459327787e-01 3.50984847622504703146262272639433e-01 6.16384778464870676170650476421997e-04
            6.45494417367507011817906459327787e-01 3.52073500998824817295740352562916e-03 6.16384778464870676170650476421997e-04
            4.20870827625693666451667240835377e-01 5.98294728636777515748401690132141e-04 2.24505119098073640301199693780632e-04
            4.20870827625693666451667240835377e-01 5.78530877645669572295616944757057e-01 2.24505119098073640301199693780632e-04
            5.98294728636777515748401690132141e-04 4.20870827625693666451667240835377e-01 2.24505119098073640301199693780632e-04
            5.98294728636777515748401690132141e-04 5.78530877645669572295616944757057e-01 2.24505119098073640301199693780632e-04
            5.78530877645669572295616944757057e-01 4.20870827625693666451667240835377e-01 2.24505119098073640301199693780632e-04
            5.78530877645669572295616944757057e-01 5.98294728636777515748401690132141e-04 2.24505119098073640301199693780632e-04
            2.52656038560264695203017026869929e-01 9.96837194601994935894140326126944e-02 2.84409163267837408056726289373728e-03
            2.52656038560264695203017026869929e-01 6.47660241979535755696417709259549e-01 2.84409163267837408056726289373728e-03
            9.96837194601994935894140326126944e-02 2.52656038560264695203017026869929e-01 2.84409163267837408056726289373728e-03
            9.96837194601994935894140326126944e-02 6.47660241979535755696417709259549e-01 2.84409163267837408056726289373728e-03
            6.47660241979535755696417709259549e-01 2.52656038560264695203017026869929e-01 2.84409163267837408056726289373728e-03
            6.47660241979535755696417709259549e-01 9.96837194601994935894140326126944e-02 2.84409163267837408056726289373728e-03
            3.01670333560527526017835953098256e-01 1.30823914375048672154022710856225e-01 3.37395444806113092156563126877700e-03
            3.01670333560527526017835953098256e-01 5.67505752064423774072565720416605e-01 3.37395444806113092156563126877700e-03
            1.30823914375048672154022710856225e-01 3.01670333560527526017835953098256e-01 3.37395444806113092156563126877700e-03
            1.30823914375048672154022710856225e-01 5.67505752064423774072565720416605e-01 3.37395444806113092156563126877700e-03
            5.67505752064423774072565720416605e-01 3.01670333560527526017835953098256e-01 3.37395444806113092156563126877700e-03
            5.67505752064423774072565720416605e-01 1.30823914375048672154022710856225e-01 3.37395444806113092156563126877700e-03
            1.60613879667907410997429451526841e-02 2.99558774001274126519422935643888e-03 1.39280312524316953436132049404250e-04
            1.60613879667907410997429451526841e-02 9.80943024293196463858635070209857e-01 1.39280312524316953436132049404250e-04
            2.99558774001274126519422935643888e-03 1.60613879667907410997429451526841e-02 1.39280312524316953436132049404250e-04
            2.99558774001274126519422935643888e-03 9.80943024293196463858635070209857e-01 1.39280312524316953436132049404250e-04
            9.80943024293196463858635070209857e-01 1.60613879667907410997429451526841e-02 1.39280312524316953436132049404250e-04
            9.80943024293196463858635070209857e-01 2.99558774001274126519422935643888e-03 1.39280312524316953436132049404250e-04
            ];
        
    case  41
        
        % ALG. DEG.:   41
        % PTS CARD.:  309
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.5e-15
        
        xyw=[
            1.81005509308153171577071560705008e-03 1.81005509308153171577071560705008e-03 2.61046615627579676728684671127567e-05
            1.81005509308153171577071560705008e-03 9.96379889813836960854587232461199e-01 2.61046615627579676728684671127567e-05
            9.96379889813836960854587232461199e-01 1.81005509308153171577071560705008e-03 2.61046615627579676728684671127567e-05
            3.14607751383877753958984158089152e-01 3.14607751383877753958984158089152e-01 3.22163483544135034394861882844907e-03
            3.14607751383877753958984158089152e-01 3.70784497232244492082031683821697e-01 3.22163483544135034394861882844907e-03
            3.70784497232244492082031683821697e-01 3.14607751383877753958984158089152e-01 3.22163483544135034394861882844907e-03
            3.71832457839435748248035906726727e-01 3.71832457839435748248035906726727e-01 2.82310188625230608708283064345324e-03
            3.71832457839435748248035906726727e-01 2.56335084321128503503928186546545e-01 2.82310188625230608708283064345324e-03
            2.56335084321128503503928186546545e-01 3.71832457839435748248035906726727e-01 2.82310188625230608708283064345324e-03
            4.81848960175834395425198408702272e-01 4.81848960175834395425198408702272e-01 1.48755341237118664243777033107108e-03
            4.81848960175834395425198408702272e-01 3.63020796483312091496031825954560e-02 1.48755341237118664243777033107108e-03
            3.63020796483312091496031825954560e-02 4.81848960175834395425198408702272e-01 1.48755341237118664243777033107108e-03
            1.42281998308420726262468747336243e-01 1.42281998308420726262468747336243e-01 2.12247788855559650633697543753442e-03
            1.42281998308420726262468747336243e-01 7.15436003383158602986213736585341e-01 2.12247788855559650633697543753442e-03
            7.15436003383158602986213736585341e-01 1.42281998308420726262468747336243e-01 2.12247788855559650633697543753442e-03
            1.74633614866341574378694190272654e-01 1.74633614866341574378694190272654e-01 2.51812585448819240691031851042681e-03
            1.74633614866341574378694190272654e-01 6.50732770267316906753762850712519e-01 2.51812585448819240691031851042681e-03
            6.50732770267316906753762850712519e-01 1.74633614866341574378694190272654e-01 2.51812585448819240691031851042681e-03
            4.25198078415139901764518981508445e-01 4.25198078415139901764518981508445e-01 3.00505023858111926585157114288904e-03
            4.25198078415139901764518981508445e-01 1.49603843169720196470962036983110e-01 3.00505023858111926585157114288904e-03
            1.49603843169720196470962036983110e-01 4.25198078415139901764518981508445e-01 3.00505023858111926585157114288904e-03
            2.09977372337917744804514086354175e-01 2.09977372337917744804514086354175e-01 3.03861449848868113432498461179421e-03
            2.09977372337917744804514086354175e-01 5.80045255324164510390971827291651e-01 3.03861449848868113432498461179421e-03
            5.80045255324164510390971827291651e-01 2.09977372337917744804514086354175e-01 3.03861449848868113432498461179421e-03
            2.62739973625342737584276164852781e-01 2.62739973625342737584276164852781e-01 3.38357010421123536800247499911620e-03
            2.62739973625342737584276164852781e-01 4.74520052749314524831447670294438e-01 3.38357010421123536800247499911620e-03
            4.74520052749314524831447670294438e-01 2.62739973625342737584276164852781e-01 3.38357010421123536800247499911620e-03
            4.98603639231114337260208912994131e-01 4.98603639231114337260208912994131e-01 5.11819985150216666229538997612281e-04
            4.98603639231114337260208912994131e-01 2.79272153777132547958217401173897e-03 5.11819985150216666229538997612281e-04
            2.79272153777132547958217401173897e-03 4.98603639231114337260208912994131e-01 5.11819985150216666229538997612281e-04
            3.99566525526970250936642514716368e-01 3.99566525526970250936642514716368e-01 3.61505486647584802634147749245130e-03
            3.99566525526970250936642514716368e-01 2.00866948946059498126714970567264e-01 3.61505486647584802634147749245130e-03
            2.00866948946059498126714970567264e-01 3.99566525526970250936642514716368e-01 3.61505486647584802634147749245130e-03
            4.66761057785514965612350124501972e-01 4.66761057785514965612350124501972e-01 2.56693516261922256502248096410312e-03
            4.66761057785514965612350124501972e-01 6.64778844289700687752997509960551e-02 2.56693516261922256502248096410312e-03
            6.64778844289700687752997509960551e-02 4.66761057785514965612350124501972e-01 2.56693516261922256502248096410312e-03
            1.03095485576030487795406997975078e-01 1.03095485576030487795406997975078e-01 1.98478451967980555267767606153484e-03
            1.03095485576030487795406997975078e-01 7.93809028847939024409186004049843e-01 1.98478451967980555267767606153484e-03
            7.93809028847939024409186004049843e-01 1.03095485576030487795406997975078e-01 1.98478451967980555267767606153484e-03
            9.33847895869260646861675922991708e-01 2.67728022087155327624063971825308e-02 4.73316998226449626475942356051974e-04
            9.33847895869260646861675922991708e-01 3.93793019220238238453646317793755e-02 4.73316998226449626475942356051974e-04
            2.67728022087155327624063971825308e-02 9.33847895869260646861675922991708e-01 4.73316998226449626475942356051974e-04
            2.67728022087155327624063971825308e-02 3.93793019220238238453646317793755e-02 4.73316998226449626475942356051974e-04
            3.93793019220238238453646317793755e-02 9.33847895869260646861675922991708e-01 4.73316998226449626475942356051974e-04
            3.93793019220238238453646317793755e-02 2.67728022087155327624063971825308e-02 4.73316998226449626475942356051974e-04
            7.08749727266859141927568543906091e-02 2.14673861123130318984131292836537e-02 5.38919164693541581818980734652769e-04
            7.08749727266859141927568543906091e-02 9.07657641161001005336572688975139e-01 5.38919164693541581818980734652769e-04
            2.14673861123130318984131292836537e-02 7.08749727266859141927568543906091e-02 5.38919164693541581818980734652769e-04
            2.14673861123130318984131292836537e-02 9.07657641161001005336572688975139e-01 5.38919164693541581818980734652769e-04
            9.07657641161001005336572688975139e-01 7.08749727266859141927568543906091e-02 5.38919164693541581818980734652769e-04
            9.07657641161001005336572688975139e-01 2.14673861123130318984131292836537e-02 5.38919164693541581818980734652769e-04
            2.27676589099523561665616711024995e-02 1.19851112427548141953215221633400e-02 2.76775703152319619247678783580113e-04
            2.27676589099523561665616711024995e-02 9.65247229847292831372840282710968e-01 2.76775703152319619247678783580113e-04
            1.19851112427548141953215221633400e-02 2.27676589099523561665616711024995e-02 2.76775703152319619247678783580113e-04
            1.19851112427548141953215221633400e-02 9.65247229847292831372840282710968e-01 2.76775703152319619247678783580113e-04
            9.65247229847292831372840282710968e-01 2.27676589099523561665616711024995e-02 2.76775703152319619247678783580113e-04
            9.65247229847292831372840282710968e-01 1.19851112427548141953215221633400e-02 2.76775703152319619247678783580113e-04
            2.12702658519097947964482386851159e-01 9.64112026242689990063183813617798e-02 1.87236045763998363092306398414166e-03
            2.12702658519097947964482386851159e-01 6.90886138856632969762472384900320e-01 1.87236045763998363092306398414166e-03
            9.64112026242689990063183813617798e-02 2.12702658519097947964482386851159e-01 1.87236045763998363092306398414166e-03
            9.64112026242689990063183813617798e-02 6.90886138856632969762472384900320e-01 1.87236045763998363092306398414166e-03
            6.90886138856632969762472384900320e-01 2.12702658519097947964482386851159e-01 1.87236045763998363092306398414166e-03
            6.90886138856632969762472384900320e-01 9.64112026242689990063183813617798e-02 1.87236045763998363092306398414166e-03
            1.97079994941520175721905161481118e-01 1.27978222846432787562775956757832e-01 2.03686960481117831575637033836301e-03
            1.97079994941520175721905161481118e-01 6.74941782212047036715318881761050e-01 2.03686960481117831575637033836301e-03
            1.27978222846432787562775956757832e-01 1.97079994941520175721905161481118e-01 2.03686960481117831575637033836301e-03
            1.27978222846432787562775956757832e-01 6.74941782212047036715318881761050e-01 2.03686960481117831575637033836301e-03
            6.74941782212047036715318881761050e-01 1.97079994941520175721905161481118e-01 2.03686960481117831575637033836301e-03
            6.74941782212047036715318881761050e-01 1.27978222846432787562775956757832e-01 2.03686960481117831575637033836301e-03
            6.39130414360007115259776355742360e-02 4.23991066111561523688244790264434e-02 8.94377285332313351484678776159853e-04
            6.39130414360007115259776355742360e-02 8.93687851952843059777364942419808e-01 8.94377285332313351484678776159853e-04
            4.23991066111561523688244790264434e-02 6.39130414360007115259776355742360e-02 8.94377285332313351484678776159853e-04
            4.23991066111561523688244790264434e-02 8.93687851952843059777364942419808e-01 8.94377285332313351484678776159853e-04
            8.93687851952843059777364942419808e-01 6.39130414360007115259776355742360e-02 8.94377285332313351484678776159853e-04
            8.93687851952843059777364942419808e-01 4.23991066111561523688244790264434e-02 8.94377285332313351484678776159853e-04
            9.16254632757576847890135240959353e-02 1.16869974011746882713858042279753e-02 5.15712369071734746950319561165088e-04
            9.16254632757576847890135240959353e-02 8.96687539323067550611767728696577e-01 5.15712369071734746950319561165088e-04
            1.16869974011746882713858042279753e-02 9.16254632757576847890135240959353e-02 5.15712369071734746950319561165088e-04
            1.16869974011746882713858042279753e-02 8.96687539323067550611767728696577e-01 5.15712369071734746950319561165088e-04
            8.96687539323067550611767728696577e-01 9.16254632757576847890135240959353e-02 5.15712369071734746950319561165088e-04
            8.96687539323067550611767728696577e-01 1.16869974011746882713858042279753e-02 5.15712369071734746950319561165088e-04
            8.89175403557379839813989974572905e-02 6.32415329385831864739131447095133e-02 1.26610363416353199385311345537275e-03
            8.89175403557379839813989974572905e-02 8.47840926705678898933626896905480e-01 1.26610363416353199385311345537275e-03
            6.32415329385831864739131447095133e-02 8.89175403557379839813989974572905e-02 1.26610363416353199385311345537275e-03
            6.32415329385831864739131447095133e-02 8.47840926705678898933626896905480e-01 1.26610363416353199385311345537275e-03
            8.47840926705678898933626896905480e-01 8.89175403557379839813989974572905e-02 1.26610363416353199385311345537275e-03
            8.47840926705678898933626896905480e-01 6.32415329385831864739131447095133e-02 1.26610363416353199385311345537275e-03
            4.59230558903699015438348851603223e-01 1.48409448407795345470372438967388e-02 1.12623927896231767163681691812371e-03
            4.59230558903699015438348851603223e-01 5.25928496255521404911803529103054e-01 1.12623927896231767163681691812371e-03
            1.48409448407795345470372438967388e-02 4.59230558903699015438348851603223e-01 1.12623927896231767163681691812371e-03
            1.48409448407795345470372438967388e-02 5.25928496255521404911803529103054e-01 1.12623927896231767163681691812371e-03
            5.25928496255521404911803529103054e-01 4.59230558903699015438348851603223e-01 1.12623927896231767163681691812371e-03
            5.25928496255521404911803529103054e-01 1.48409448407795345470372438967388e-02 1.12623927896231767163681691812371e-03
            3.38843822348438572511497568484629e-02 1.84003961561307214853111524632823e-03 1.40918754232970182228498745580225e-04
            3.38843822348438572511497568484629e-02 9.64275578149543122208342538215220e-01 1.40918754232970182228498745580225e-04
            1.84003961561307214853111524632823e-03 3.38843822348438572511497568484629e-02 1.40918754232970182228498745580225e-04
            1.84003961561307214853111524632823e-03 9.64275578149543122208342538215220e-01 1.40918754232970182228498745580225e-04
            9.64275578149543122208342538215220e-01 3.38843822348438572511497568484629e-02 1.40918754232970182228498745580225e-04
            9.64275578149543122208342538215220e-01 1.84003961561307214853111524632823e-03 1.40918754232970182228498745580225e-04
            3.20075210406615351388381895958446e-01 2.58057113739871668567360529777943e-01 3.05468653482410821067838391229543e-03
            3.20075210406615351388381895958446e-01 4.21867675853512980044257574263611e-01 3.05468653482410821067838391229543e-03
            2.58057113739871668567360529777943e-01 3.20075210406615351388381895958446e-01 3.05468653482410821067838391229543e-03
            2.58057113739871668567360529777943e-01 4.21867675853512980044257574263611e-01 3.05468653482410821067838391229543e-03
            4.21867675853512980044257574263611e-01 3.20075210406615351388381895958446e-01 3.05468653482410821067838391229543e-03
            4.21867675853512980044257574263611e-01 2.58057113739871668567360529777943e-01 3.05468653482410821067838391229543e-03
            4.22068492531888939112150183063932e-01 3.62192254340186342576046740759921e-02 1.65448560765117679872970857246628e-03
            4.22068492531888939112150183063932e-01 5.41712282034092384996881719416706e-01 1.65448560765117679872970857246628e-03
            3.62192254340186342576046740759921e-02 4.22068492531888939112150183063932e-01 1.65448560765117679872970857246628e-03
            3.62192254340186342576046740759921e-02 5.41712282034092384996881719416706e-01 1.65448560765117679872970857246628e-03
            5.41712282034092384996881719416706e-01 4.22068492531888939112150183063932e-01 1.65448560765117679872970857246628e-03
            5.41712282034092384996881719416706e-01 3.62192254340186342576046740759921e-02 1.65448560765117679872970857246628e-03
            4.78764410729283981793535929227801e-02 1.05439660443664238315131242984535e-02 3.75170187050299106055334386056188e-04
            4.78764410729283981793535929227801e-02 9.41579592882705096457129911868833e-01 3.75170187050299106055334386056188e-04
            1.05439660443664238315131242984535e-02 4.78764410729283981793535929227801e-02 3.75170187050299106055334386056188e-04
            1.05439660443664238315131242984535e-02 9.41579592882705096457129911868833e-01 3.75170187050299106055334386056188e-04
            9.41579592882705096457129911868833e-01 4.78764410729283981793535929227801e-02 3.75170187050299106055334386056188e-04
            9.41579592882705096457129911868833e-01 1.05439660443664238315131242984535e-02 3.75170187050299106055334386056188e-04
            3.61799022706076223165894134581322e-01 1.51026887499565554140446010933374e-01 3.11182961660478674473240268127938e-03
            3.61799022706076223165894134581322e-01 4.87174089794358222693659854485304e-01 3.11182961660478674473240268127938e-03
            1.51026887499565554140446010933374e-01 3.61799022706076223165894134581322e-01 3.11182961660478674473240268127938e-03
            1.51026887499565554140446010933374e-01 4.87174089794358222693659854485304e-01 3.11182961660478674473240268127938e-03
            4.87174089794358222693659854485304e-01 3.61799022706076223165894134581322e-01 3.11182961660478674473240268127938e-03
            4.87174089794358222693659854485304e-01 1.51026887499565554140446010933374e-01 3.11182961660478674473240268127938e-03
            6.84569430922209731793515175013454e-02 2.48206090173236360030273317534011e-03 2.45210435507164665330792630015821e-04
            6.84569430922209731793515175013454e-02 9.29060996006046746487072596210055e-01 2.45210435507164665330792630015821e-04
            2.48206090173236360030273317534011e-03 6.84569430922209731793515175013454e-02 2.45210435507164665330792630015821e-04
            2.48206090173236360030273317534011e-03 9.29060996006046746487072596210055e-01 2.45210435507164665330792630015821e-04
            9.29060996006046746487072596210055e-01 6.84569430922209731793515175013454e-02 2.45210435507164665330792630015821e-04
            9.29060996006046746487072596210055e-01 2.48206090173236360030273317534011e-03 2.45210435507164665330792630015821e-04
            2.36288472596456022856159506773110e-01 1.54957767021962872222573537328572e-01 2.94532213285211624462234247801007e-03
            2.36288472596456022856159506773110e-01 6.08753760381581132676842571527231e-01 2.94532213285211624462234247801007e-03
            1.54957767021962872222573537328572e-01 2.36288472596456022856159506773110e-01 2.94532213285211624462234247801007e-03
            1.54957767021962872222573537328572e-01 6.08753760381581132676842571527231e-01 2.94532213285211624462234247801007e-03
            6.08753760381581132676842571527231e-01 2.36288472596456022856159506773110e-01 2.94532213285211624462234247801007e-03
            6.08753760381581132676842571527231e-01 1.54957767021962872222573537328572e-01 2.94532213285211624462234247801007e-03
            2.97895418985049564852829462324735e-01 1.51961553665098864085436503046367e-01 3.11906005955700157961096863346029e-03
            2.97895418985049564852829462324735e-01 5.50143027349851543306158418999985e-01 3.11906005955700157961096863346029e-03
            1.51961553665098864085436503046367e-01 2.97895418985049564852829462324735e-01 3.11906005955700157961096863346029e-03
            1.51961553665098864085436503046367e-01 5.50143027349851543306158418999985e-01 3.11906005955700157961096863346029e-03
            5.50143027349851543306158418999985e-01 2.97895418985049564852829462324735e-01 3.11906005955700157961096863346029e-03
            5.50143027349851543306158418999985e-01 1.51961553665098864085436503046367e-01 3.11906005955700157961096863346029e-03
            1.12557251523921778013637151616422e-01 2.37754573037215425485024766771858e-03 2.94459094064066602067231137596082e-04
            1.12557251523921778013637151616422e-01 8.85065202745705992271041395724751e-01 2.94459094064066602067231137596082e-04
            2.37754573037215425485024766771858e-03 1.12557251523921778013637151616422e-01 2.94459094064066602067231137596082e-04
            2.37754573037215425485024766771858e-03 8.85065202745705992271041395724751e-01 2.94459094064066602067231137596082e-04
            8.85065202745705992271041395724751e-01 1.12557251523921778013637151616422e-01 2.94459094064066602067231137596082e-04
            8.85065202745705992271041395724751e-01 2.37754573037215425485024766771858e-03 2.94459094064066602067231137596082e-04
            1.20658629713076021722217845422165e-02 3.04923808583817523149006589733290e-03 1.19525780902384455117552775060830e-04
            1.20658629713076021722217845422165e-02 9.84884898942854247749778551224153e-01 1.19525780902384455117552775060830e-04
            3.04923808583817523149006589733290e-03 1.20658629713076021722217845422165e-02 1.19525780902384455117552775060830e-04
            3.04923808583817523149006589733290e-03 9.84884898942854247749778551224153e-01 1.19525780902384455117552775060830e-04
            9.84884898942854247749778551224153e-01 1.20658629713076021722217845422165e-02 1.19525780902384455117552775060830e-04
            9.84884898942854247749778551224153e-01 3.04923808583817523149006589733290e-03 1.19525780902384455117552775060830e-04
            3.90653097086316902775138260039967e-01 1.50754657505242934539824162243349e-02 1.16906687993455953369914546158270e-03
            3.90653097086316902775138260039967e-01 5.94271437163158777750027184083592e-01 1.16906687993455953369914546158270e-03
            1.50754657505242934539824162243349e-02 3.90653097086316902775138260039967e-01 1.16906687993455953369914546158270e-03
            1.50754657505242934539824162243349e-02 5.94271437163158777750027184083592e-01 1.16906687993455953369914546158270e-03
            5.94271437163158777750027184083592e-01 3.90653097086316902775138260039967e-01 1.16906687993455953369914546158270e-03
            5.94271437163158777750027184083592e-01 1.50754657505242934539824162243349e-02 1.16906687993455953369914546158270e-03
            4.27410727373970178444295697772759e-01 2.88455410756848591377621282560995e-03 5.22716485387803410714302465578385e-04
            4.27410727373970178444295697772759e-01 5.69704718518461272758202085242374e-01 5.22716485387803410714302465578385e-04
            2.88455410756848591377621282560995e-03 4.27410727373970178444295697772759e-01 5.22716485387803410714302465578385e-04
            2.88455410756848591377621282560995e-03 5.69704718518461272758202085242374e-01 5.22716485387803410714302465578385e-04
            5.69704718518461272758202085242374e-01 4.27410727373970178444295697772759e-01 5.22716485387803410714302465578385e-04
            5.69704718518461272758202085242374e-01 2.88455410756848591377621282560995e-03 5.22716485387803410714302465578385e-04
            4.11845522097719740628463114262559e-01 1.04765137725845669525170933411573e-01 2.99509456015477782547806739898988e-03
            4.11845522097719740628463114262559e-01 4.83389340176434589846365952325868e-01 2.99509456015477782547806739898988e-03
            1.04765137725845669525170933411573e-01 4.11845522097719740628463114262559e-01 2.99509456015477782547806739898988e-03
            1.04765137725845669525170933411573e-01 4.83389340176434589846365952325868e-01 2.99509456015477782547806739898988e-03
            4.83389340176434589846365952325868e-01 4.11845522097719740628463114262559e-01 2.99509456015477782547806739898988e-03
            4.83389340176434589846365952325868e-01 1.04765137725845669525170933411573e-01 2.99509456015477782547806739898988e-03
            3.22221194516843667532413064691355e-01 1.48317675755721811031717649598249e-02 1.14412479363062567945530378921148e-03
            3.22221194516843667532413064691355e-01 6.62947037907584246774206349073211e-01 1.14412479363062567945530378921148e-03
            1.48317675755721811031717649598249e-02 3.22221194516843667532413064691355e-01 1.14412479363062567945530378921148e-03
            1.48317675755721811031717649598249e-02 6.62947037907584246774206349073211e-01 1.14412479363062567945530378921148e-03
            6.62947037907584246774206349073211e-01 3.22221194516843667532413064691355e-01 1.14412479363062567945530378921148e-03
            6.62947037907584246774206349073211e-01 1.48317675755721811031717649598249e-02 1.14412479363062567945530378921148e-03
            3.54524901630881295044872558719362e-01 3.67451935646271599100742832888500e-02 1.80524352827792118614258498610070e-03
            3.54524901630881295044872558719362e-01 6.08729904804491517289477542362874e-01 1.80524352827792118614258498610070e-03
            3.67451935646271599100742832888500e-02 3.54524901630881295044872558719362e-01 1.80524352827792118614258498610070e-03
            3.67451935646271599100742832888500e-02 6.08729904804491517289477542362874e-01 1.80524352827792118614258498610070e-03
            6.08729904804491517289477542362874e-01 3.54524901630881295044872558719362e-01 1.80524352827792118614258498610070e-03
            6.08729904804491517289477542362874e-01 3.67451935646271599100742832888500e-02 1.80524352827792118614258498610070e-03
            3.32027617298779720300672124722041e-01 2.02832977056254276648772361113515e-01 3.52932890427508144312596094493983e-03
            3.32027617298779720300672124722041e-01 4.65139405644966030806131129793357e-01 3.52932890427508144312596094493983e-03
            2.02832977056254276648772361113515e-01 3.32027617298779720300672124722041e-01 3.52932890427508144312596094493983e-03
            2.02832977056254276648772361113515e-01 4.65139405644966030806131129793357e-01 3.52932890427508144312596094493983e-03
            4.65139405644966030806131129793357e-01 3.32027617298779720300672124722041e-01 3.52932890427508144312596094493983e-03
            4.65139405644966030806131129793357e-01 2.02832977056254276648772361113515e-01 3.52932890427508144312596094493983e-03
            2.53029739084772276047630157336243e-01 6.55893176896967106115354795292660e-02 2.18927628685137770289670378076607e-03
            2.53029739084772276047630157336243e-01 6.81380943225531110485349017835688e-01 2.18927628685137770289670378076607e-03
            6.55893176896967106115354795292660e-02 2.53029739084772276047630157336243e-01 2.18927628685137770289670378076607e-03
            6.55893176896967106115354795292660e-02 6.81380943225531110485349017835688e-01 2.18927628685137770289670378076607e-03
            6.81380943225531110485349017835688e-01 2.53029739084772276047630157336243e-01 2.18927628685137770289670378076607e-03
            6.81380943225531110485349017835688e-01 6.55893176896967106115354795292660e-02 2.18927628685137770289670378076607e-03
            3.57039904894509707578009738426772e-01 2.83340788181253838001749656427819e-03 5.08569179475153979215484856268858e-04
            3.57039904894509707578009738426772e-01 6.40126687223677826032997018046444e-01 5.08569179475153979215484856268858e-04
            2.83340788181253838001749656427819e-03 3.57039904894509707578009738426772e-01 5.08569179475153979215484856268858e-04
            2.83340788181253838001749656427819e-03 6.40126687223677826032997018046444e-01 5.08569179475153979215484856268858e-04
            6.40126687223677826032997018046444e-01 3.57039904894509707578009738426772e-01 5.08569179475153979215484856268858e-04
            6.40126687223677826032997018046444e-01 2.83340788181253838001749656427819e-03 5.08569179475153979215484856268858e-04
            2.87060689211655395425282222277019e-01 3.60525157666744186513163583640562e-02 1.72774226236022945477066592445681e-03
            2.87060689211655395425282222277019e-01 6.76886795021670151228931899822783e-01 1.72774226236022945477066592445681e-03
            3.60525157666744186513163583640562e-02 2.87060689211655395425282222277019e-01 1.72774226236022945477066592445681e-03
            3.60525157666744186513163583640562e-02 6.76886795021670151228931899822783e-01 1.72774226236022945477066592445681e-03
            6.76886795021670151228931899822783e-01 2.87060689211655395425282222277019e-01 1.72774226236022945477066592445681e-03
            6.76886795021670151228931899822783e-01 3.60525157666744186513163583640562e-02 1.72774226236022945477066592445681e-03
            1.36131351631136071622663052949065e-01 1.41561971271275018058322103797764e-02 8.31309620788596328848363281593947e-04
            1.36131351631136071622663052949065e-01 8.49712451241736443918739496439230e-01 8.31309620788596328848363281593947e-04
            1.41561971271275018058322103797764e-02 1.36131351631136071622663052949065e-01 8.31309620788596328848363281593947e-04
            1.41561971271275018058322103797764e-02 8.49712451241736443918739496439230e-01 8.31309620788596328848363281593947e-04
            8.49712451241736443918739496439230e-01 1.36131351631136071622663052949065e-01 8.31309620788596328848363281593947e-04
            8.49712451241736443918739496439230e-01 1.41561971271275018058322103797764e-02 8.31309620788596328848363281593947e-04
            1.32224125661775021578492328444554e-01 6.37728096721327936124978918996931e-02 1.67558532663434852673367547737371e-03
            1.32224125661775021578492328444554e-01 8.04003064666092170931221971841296e-01 1.67558532663434852673367547737371e-03
            6.37728096721327936124978918996931e-02 1.32224125661775021578492328444554e-01 1.67558532663434852673367547737371e-03
            6.37728096721327936124978918996931e-02 8.04003064666092170931221971841296e-01 1.67558532663434852673367547737371e-03
            8.04003064666092170931221971841296e-01 1.32224125661775021578492328444554e-01 1.67558532663434852673367547737371e-03
            8.04003064666092170931221971841296e-01 6.37728096721327936124978918996931e-02 1.67558532663434852673367547737371e-03
            1.07924219839552332222964992070047e-01 3.35534395458731288242049117798160e-02 1.19328232526644479119526387478345e-03
            1.07924219839552332222964992070047e-01 8.58522340614574552830617903964594e-01 1.19328232526644479119526387478345e-03
            3.35534395458731288242049117798160e-02 1.07924219839552332222964992070047e-01 1.19328232526644479119526387478345e-03
            3.35534395458731288242049117798160e-02 8.58522340614574552830617903964594e-01 1.19328232526644479119526387478345e-03
            8.58522340614574552830617903964594e-01 1.07924219839552332222964992070047e-01 1.19328232526644479119526387478345e-03
            8.58522340614574552830617903964594e-01 3.35534395458731288242049117798160e-02 1.19328232526644479119526387478345e-03
            1.52886902141386588072080598976754e-01 9.95790446035427906368298067718570e-02 2.08768949658442049682749441785745e-03
            1.52886902141386588072080598976754e-01 7.47534053255070607413301786436932e-01 2.08768949658442049682749441785745e-03
            9.95790446035427906368298067718570e-02 1.52886902141386588072080598976754e-01 2.08768949658442049682749441785745e-03
            9.95790446035427906368298067718570e-02 7.47534053255070607413301786436932e-01 2.08768949658442049682749441785745e-03
            7.47534053255070607413301786436932e-01 1.52886902141386588072080598976754e-01 2.08768949658442049682749441785745e-03
            7.47534053255070607413301786436932e-01 9.95790446035427906368298067718570e-02 2.08768949658442049682749441785745e-03
            1.88101771929606642608234778890619e-01 6.46491520250393747470951666400651e-02 1.93556074427131776828636677834083e-03
            1.88101771929606642608234778890619e-01 7.47249076045353954889094438840402e-01 1.93556074427131776828636677834083e-03
            6.46491520250393747470951666400651e-02 1.88101771929606642608234778890619e-01 1.93556074427131776828636677834083e-03
            6.46491520250393747470951666400651e-02 7.47249076045353954889094438840402e-01 1.93556074427131776828636677834083e-03
            7.47249076045353954889094438840402e-01 1.88101771929606642608234778890619e-01 1.93556074427131776828636677834083e-03
            7.47249076045353954889094438840402e-01 6.46491520250393747470951666400651e-02 1.93556074427131776828636677834083e-03
            2.54982429150887923707813342844020e-01 1.48987268708079909768082771392983e-02 1.09616959802194720297041641288160e-03
            2.54982429150887923707813342844020e-01 7.30118843978304066233420144271804e-01 1.09616959802194720297041641288160e-03
            1.48987268708079909768082771392983e-02 2.54982429150887923707813342844020e-01 1.09616959802194720297041641288160e-03
            1.48987268708079909768082771392983e-02 7.30118843978304066233420144271804e-01 1.09616959802194720297041641288160e-03
            7.30118843978304066233420144271804e-01 2.54982429150887923707813342844020e-01 1.09616959802194720297041641288160e-03
            7.30118843978304066233420144271804e-01 1.48987268708079909768082771392983e-02 1.09616959802194720297041641288160e-03
            2.23078080481277696289765799519955e-01 2.85444727801385713522597242786105e-03 4.56366168260760238489098794545384e-04
            2.23078080481277696289765799519955e-01 7.74067472240708376318707450991496e-01 4.56366168260760238489098794545384e-04
            2.85444727801385713522597242786105e-03 2.23078080481277696289765799519955e-01 4.56366168260760238489098794545384e-04
            2.85444727801385713522597242786105e-03 7.74067472240708376318707450991496e-01 4.56366168260760238489098794545384e-04
            7.74067472240708376318707450991496e-01 2.23078080481277696289765799519955e-01 4.56366168260760238489098794545384e-04
            7.74067472240708376318707450991496e-01 2.85444727801385713522597242786105e-03 4.56366168260760238489098794545384e-04
            1.64050736021032539380115622407175e-01 2.83077182118177030772931956903449e-03 4.01417326307675211786302194028053e-04
            1.64050736021032539380115622407175e-01 8.33118492157785661689217704406474e-01 4.01417326307675211786302194028053e-04
            2.83077182118177030772931956903449e-03 1.64050736021032539380115622407175e-01 4.01417326307675211786302194028053e-04
            2.83077182118177030772931956903449e-03 8.33118492157785661689217704406474e-01 4.01417326307675211786302194028053e-04
            8.33118492157785661689217704406474e-01 1.64050736021032539380115622407175e-01 4.01417326307675211786302194028053e-04
            8.33118492157785661689217704406474e-01 2.83077182118177030772931956903449e-03 4.01417326307675211786302194028053e-04
            2.88151117418988578222638352599461e-01 2.83594679658896297452796986249268e-03 4.89679800789311839986917007649936e-04
            2.88151117418988578222638352599461e-01 7.09012935784422482221600603224942e-01 4.89679800789311839986917007649936e-04
            2.83594679658896297452796986249268e-03 2.88151117418988578222638352599461e-01 4.89679800789311839986917007649936e-04
            2.83594679658896297452796986249268e-03 7.09012935784422482221600603224942e-01 4.89679800789311839986917007649936e-04
            7.09012935784422482221600603224942e-01 2.88151117418988578222638352599461e-01 4.89679800789311839986917007649936e-04
            7.09012935784422482221600603224942e-01 2.83594679658896297452796986249268e-03 4.89679800789311839986917007649936e-04
            3.40957788874061529238446155432030e-01 1.05632686278177900796038102271268e-01 2.97754338044403698074202857526416e-03
            3.40957788874061529238446155432030e-01 5.53409524847760514454364511038875e-01 2.97754338044403698074202857526416e-03
            1.05632686278177900796038102271268e-01 3.40957788874061529238446155432030e-01 2.97754338044403698074202857526416e-03
            1.05632686278177900796038102271268e-01 5.53409524847760514454364511038875e-01 2.97754338044403698074202857526416e-03
            5.53409524847760514454364511038875e-01 3.40957788874061529238446155432030e-01 2.97754338044403698074202857526416e-03
            5.53409524847760514454364511038875e-01 1.05632686278177900796038102271268e-01 2.97754338044403698074202857526416e-03
            2.68293448069283901880055509536760e-01 2.05408895782746736546897636799258e-01 3.37625766935876819516137992138738e-03
            2.68293448069283901880055509536760e-01 5.26297656147969306061895622406155e-01 3.37625766935876819516137992138738e-03
            2.05408895782746736546897636799258e-01 2.68293448069283901880055509536760e-01 3.37625766935876819516137992138738e-03
            2.05408895782746736546897636799258e-01 5.26297656147969306061895622406155e-01 3.37625766935876819516137992138738e-03
            5.26297656147969306061895622406155e-01 2.68293448069283901880055509536760e-01 3.37625766935876819516137992138738e-03
            5.26297656147969306061895622406155e-01 2.05408895782746736546897636799258e-01 3.37625766935876819516137992138738e-03
            3.19409003528324686449479941074969e-01 6.74786310679263284795226240930788e-02 2.46633730412294501291303738810257e-03
            3.19409003528324686449479941074969e-01 6.13112365403749026704360858275322e-01 2.46633730412294501291303738810257e-03
            6.74786310679263284795226240930788e-02 3.19409003528324686449479941074969e-01 2.46633730412294501291303738810257e-03
            6.74786310679263284795226240930788e-02 6.13112365403749026704360858275322e-01 2.46633730412294501291303738810257e-03
            6.13112365403749026704360858275322e-01 3.19409003528324686449479941074969e-01 2.46633730412294501291303738810257e-03
            6.13112365403749026704360858275322e-01 6.74786310679263284795226240930788e-02 2.46633730412294501291303738810257e-03
            1.92295465185952280107173351098027e-01 1.49331073629515708595105749623144e-02 9.99758561822005915181721391604697e-04
            1.92295465185952280107173351098027e-01 7.92771427451096077909653558890568e-01 9.99758561822005915181721391604697e-04
            1.49331073629515708595105749623144e-02 1.92295465185952280107173351098027e-01 9.99758561822005915181721391604697e-04
            1.49331073629515708595105749623144e-02 7.92771427451096077909653558890568e-01 9.99758561822005915181721391604697e-04
            7.92771427451096077909653558890568e-01 1.92295465185952280107173351098027e-01 9.99758561822005915181721391604697e-04
            7.92771427451096077909653558890568e-01 1.49331073629515708595105749623144e-02 9.99758561822005915181721391604697e-04
            1.60414053402923612035024802935368e-01 3.53852934054426707111140615324985e-02 1.42908494641375569469965434166170e-03
            1.60414053402923612035024802935368e-01 8.04200653191633696437179423810448e-01 1.42908494641375569469965434166170e-03
            3.53852934054426707111140615324985e-02 1.60414053402923612035024802935368e-01 1.42908494641375569469965434166170e-03
            3.53852934054426707111140615324985e-02 8.04200653191633696437179423810448e-01 1.42908494641375569469965434166170e-03
            8.04200653191633696437179423810448e-01 1.60414053402923612035024802935368e-01 1.42908494641375569469965434166170e-03
            8.04200653191633696437179423810448e-01 3.53852934054426707111140615324985e-02 1.42908494641375569469965434166170e-03
            2.21253498980421736685997302629403e-01 3.62519958338577280576586758797930e-02 1.62869147472470966568625083681354e-03
            2.21253498980421736685997302629403e-01 7.42494505185720465867404982418520e-01 1.62869147472470966568625083681354e-03
            3.62519958338577280576586758797930e-02 2.21253498980421736685997302629403e-01 1.62869147472470966568625083681354e-03
            3.62519958338577280576586758797930e-02 7.42494505185720465867404982418520e-01 1.62869147472470966568625083681354e-03
            7.42494505185720465867404982418520e-01 2.21253498980421736685997302629403e-01 1.62869147472470966568625083681354e-03
            7.42494505185720465867404982418520e-01 3.62519958338577280576586758797930e-02 1.62869147472470966568625083681354e-03
            2.71194224068813549699541454174323e-01 1.06599696973037380565152432154719e-01 2.93614082918626203080880721074664e-03
            2.71194224068813549699541454174323e-01 6.22206078958149055857518305856502e-01 2.93614082918626203080880721074664e-03
            1.06599696973037380565152432154719e-01 2.71194224068813549699541454174323e-01 2.93614082918626203080880721074664e-03
            1.06599696973037380565152432154719e-01 6.22206078958149055857518305856502e-01 2.93614082918626203080880721074664e-03
            6.22206078958149055857518305856502e-01 2.71194224068813549699541454174323e-01 2.93614082918626203080880721074664e-03
            6.22206078958149055857518305856502e-01 1.06599696973037380565152432154719e-01 2.93614082918626203080880721074664e-03
            3.92323411299770385962659702272504e-01 6.65697076536746201735894601370092e-02 2.54753922375026108093631904694121e-03
            3.92323411299770385962659702272504e-01 5.41106881046554910597023990703747e-01 2.54753922375026108093631904694121e-03
            6.65697076536746201735894601370092e-02 3.92323411299770385962659702272504e-01 2.54753922375026108093631904694121e-03
            6.65697076536746201735894601370092e-02 5.41106881046554910597023990703747e-01 2.54753922375026108093631904694121e-03
            5.41106881046554910597023990703747e-01 3.92323411299770385962659702272504e-01 2.54753922375026108093631904694121e-03
            5.41106881046554910597023990703747e-01 6.65697076536746201735894601370092e-02 2.54753922375026108093631904694121e-03
            ];
        
    case  42
        
        % ALG. DEG.:   42
        % PTS CARD.:  324
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.4e-15
        
        xyw=[
            4.02371925109793249220047073322348e-01 4.02371925109793249220047073322348e-01 1.77322000798858401757196467940503e-03
            4.02371925109793249220047073322348e-01 1.95256149780413501559905853355303e-01 1.77322000798858401757196467940503e-03
            1.95256149780413501559905853355303e-01 4.02371925109793249220047073322348e-01 1.77322000798858401757196467940503e-03
            1.36539257595404031446484349743287e-03 1.36539257595404031446484349743287e-03 1.73597677182810516765230912650253e-05
            1.36539257595404031446484349743287e-03 9.97269214848091944958241583663039e-01 1.73597677182810516765230912650253e-05
            9.97269214848091944958241583663039e-01 1.36539257595404031446484349743287e-03 1.73597677182810516765230912650253e-05
            4.97531729625205365596940509931301e-01 4.97531729625205365596940509931301e-01 5.07084821416368069189883893699289e-04
            4.97531729625205365596940509931301e-01 4.93654074958926880611898013739847e-03 5.07084821416368069189883893699289e-04
            4.93654074958926880611898013739847e-03 4.97531729625205365596940509931301e-01 5.07084821416368069189883893699289e-04
            4.79900295754419115024091979648801e-01 4.79900295754419115024091979648801e-01 1.42209297850611989834368475982274e-03
            4.79900295754419115024091979648801e-01 4.01994084911617699518160407023970e-02 1.42209297850611989834368475982274e-03
            4.01994084911617699518160407023970e-02 4.79900295754419115024091979648801e-01 1.42209297850611989834368475982274e-03
            6.64663290557835906291472838347545e-02 6.64663290557835906291472838347545e-02 9.35490769011440006716617823201432e-04
            6.64663290557835906291472838347545e-02 8.67067341888432818741705432330491e-01 9.35490769011440006716617823201432e-04
            8.67067341888432818741705432330491e-01 6.64663290557835906291472838347545e-02 9.35490769011440006716617823201432e-04
            4.32363893877459951053943143506331e-02 4.32363893877459951053943143506331e-02 6.66312500203990383008945119769351e-04
            4.32363893877459951053943143506331e-02 9.13527221224508023666999179113191e-01 6.66312500203990383008945119769351e-04
            9.13527221224508023666999179113191e-01 4.32363893877459951053943143506331e-02 6.66312500203990383008945119769351e-04
            4.90305014201646682803215071544400e-01 4.90305014201646682803215071544400e-01 1.09634084079294233209089792779878e-03
            4.90305014201646682803215071544400e-01 1.93899715967066343935698569111992e-02 1.09634084079294233209089792779878e-03
            1.93899715967066343935698569111992e-02 4.90305014201646682803215071544400e-01 1.09634084079294233209089792779878e-03
            2.69988226283729981780368234467460e-02 2.69988226283729981780368234467460e-02 4.76202977827340750548862358115798e-04
            2.69988226283729981780368234467460e-02 9.46002354743254003643926353106508e-01 4.76202977827340750548862358115798e-04
            9.46002354743254003643926353106508e-01 2.69988226283729981780368234467460e-02 4.76202977827340750548862358115798e-04
            1.38504806910322769697430089763657e-01 1.38504806910322769697430089763657e-01 2.17992502094623398847805795242039e-03
            1.38504806910322769697430089763657e-01 7.22990386179354516116291051730514e-01 2.17992502094623398847805795242039e-03
            7.22990386179354516116291051730514e-01 1.38504806910322769697430089763657e-01 2.17992502094623398847805795242039e-03
            4.67013207355763959593986101026530e-01 4.67013207355763959593986101026530e-01 2.26626612751637578416219476196147e-03
            4.67013207355763959593986101026530e-01 6.59735852884720808120277979469392e-02 2.26626612751637578416219476196147e-03
            6.59735852884720808120277979469392e-02 4.67013207355763959593986101026530e-01 2.26626612751637578416219476196147e-03
            1.06580425858858796051009676375543e-01 1.06580425858858796051009676375543e-01 1.84803533077693886779724330438057e-03
            1.06580425858858796051009676375543e-01 7.86839148282282407897980647248914e-01 1.84803533077693886779724330438057e-03
            7.86839148282282407897980647248914e-01 1.06580425858858796051009676375543e-01 1.84803533077693886779724330438057e-03
            3.11735391072587297855989163508639e-01 3.11735391072587297855989163508639e-01 4.04866880451419577230387503163911e-03
            3.11735391072587297855989163508639e-01 3.76529217854825404288021672982723e-01 4.04866880451419577230387503163911e-03
            3.76529217854825404288021672982723e-01 3.11735391072587297855989163508639e-01 4.04866880451419577230387503163911e-03
            2.44860575868881713557101420519757e-01 2.44860575868881713557101420519757e-01 3.64530052639338696812698614735382e-03
            2.44860575868881713557101420519757e-01 5.10278848262236572885797158960486e-01 3.64530052639338696812698614735382e-03
            5.10278848262236572885797158960486e-01 2.44860575868881713557101420519757e-01 3.64530052639338696812698614735382e-03
            1.74835145811190639797416679357411e-01 1.74835145811190639797416679357411e-01 3.02776339101268636075370288551767e-03
            1.74835145811190639797416679357411e-01 6.50329708377618720405166641285177e-01 3.02776339101268636075370288551767e-03
            6.50329708377618720405166641285177e-01 1.74835145811190639797416679357411e-01 3.02776339101268636075370288551767e-03
            4.27569334150190383248002490290673e-01 4.27569334150190383248002490290673e-01 3.37947231961488459006659468286671e-03
            4.27569334150190383248002490290673e-01 1.44861331699619233503995019418653e-01 3.37947231961488459006659468286671e-03
            1.44861331699619233503995019418653e-01 4.27569334150190383248002490290673e-01 3.37947231961488459006659468286671e-03
            3.74270156678447651898977710516192e-01 3.74270156678447651898977710516192e-01 3.96102228829565711915039472046374e-03
            3.74270156678447651898977710516192e-01 2.51459686643104696202044578967616e-01 3.96102228829565711915039472046374e-03
            2.51459686643104696202044578967616e-01 3.74270156678447651898977710516192e-01 3.96102228829565711915039472046374e-03
            3.30553853684353138309148789630854e-01 1.57042560146290624892295717529578e-03 2.66365847170266245320241482730239e-04
            3.30553853684353138309148789630854e-01 6.67875720714183995774249069654616e-01 2.66365847170266245320241482730239e-04
            1.57042560146290624892295717529578e-03 3.30553853684353138309148789630854e-01 2.66365847170266245320241482730239e-04
            1.57042560146290624892295717529578e-03 6.67875720714183995774249069654616e-01 2.66365847170266245320241482730239e-04
            6.67875720714183995774249069654616e-01 3.30553853684353138309148789630854e-01 2.66365847170266245320241482730239e-04
            6.67875720714183995774249069654616e-01 1.57042560146290624892295717529578e-03 2.66365847170266245320241482730239e-04
            9.69953965146683194653576265409356e-01 9.61466912071009052742631695309683e-03 2.19538513584912036574747773443050e-04
            9.69953965146683194653576265409356e-01 2.04313657326067148189974176375472e-02 2.19538513584912036574747773443050e-04
            9.61466912071009052742631695309683e-03 9.69953965146683194653576265409356e-01 2.19538513584912036574747773443050e-04
            9.61466912071009052742631695309683e-03 2.04313657326067148189974176375472e-02 2.19538513584912036574747773443050e-04
            2.04313657326067148189974176375472e-02 9.69953965146683194653576265409356e-01 2.19538513584912036574747773443050e-04
            2.04313657326067148189974176375472e-02 9.61466912071009052742631695309683e-03 2.19538513584912036574747773443050e-04
            5.71983061939756801117518136834406e-02 2.19035586156293565762709363298200e-02 5.18512523267713391732225414187951e-04
            5.71983061939756801117518136834406e-02 9.20898135190394984128658961708425e-01 5.18512523267713391732225414187951e-04
            2.19035586156293565762709363298200e-02 5.71983061939756801117518136834406e-02 5.18512523267713391732225414187951e-04
            2.19035586156293565762709363298200e-02 9.20898135190394984128658961708425e-01 5.18512523267713391732225414187951e-04
            9.20898135190394984128658961708425e-01 5.71983061939756801117518136834406e-02 5.18512523267713391732225414187951e-04
            9.20898135190394984128658961708425e-01 2.19035586156293565762709363298200e-02 5.18512523267713391732225414187951e-04
            4.53270971637945607746189580211649e-01 1.49355129029098685801424384322900e-03 2.79775975114021982492751305926504e-04
            4.53270971637945607746189580211649e-01 5.45235477071763452450170461816015e-01 2.79775975114021982492751305926504e-04
            1.49355129029098685801424384322900e-03 4.53270971637945607746189580211649e-01 2.79775975114021982492751305926504e-04
            1.49355129029098685801424384322900e-03 5.45235477071763452450170461816015e-01 2.79775975114021982492751305926504e-04
            5.45235477071763452450170461816015e-01 4.53270971637945607746189580211649e-01 2.79775975114021982492751305926504e-04
            5.45235477071763452450170461816015e-01 1.49355129029098685801424384322900e-03 2.79775975114021982492751305926504e-04
            1.40732473906720334522191251380718e-01 5.54414996743645416277246340541751e-02 1.26896025323238619865462428748515e-03
            1.40732473906720334522191251380718e-01 8.03826026418915096094508498936193e-01 1.26896025323238619865462428748515e-03
            5.54414996743645416277246340541751e-02 1.40732473906720334522191251380718e-01 1.26896025323238619865462428748515e-03
            5.54414996743645416277246340541751e-02 8.03826026418915096094508498936193e-01 1.26896025323238619865462428748515e-03
            8.03826026418915096094508498936193e-01 1.40732473906720334522191251380718e-01 1.26896025323238619865462428748515e-03
            8.03826026418915096094508498936193e-01 5.54414996743645416277246340541751e-02 1.26896025323238619865462428748515e-03
            1.83788630301941036115920269367052e-01 1.18693569421395000995644863905909e-01 2.02907889905697916971272221076106e-03
            1.83788630301941036115920269367052e-01 6.97517800276663990644010482355952e-01 2.02907889905697916971272221076106e-03
            1.18693569421395000995644863905909e-01 1.83788630301941036115920269367052e-01 2.02907889905697916971272221076106e-03
            1.18693569421395000995644863905909e-01 6.97517800276663990644010482355952e-01 2.02907889905697916971272221076106e-03
            6.97517800276663990644010482355952e-01 1.83788630301941036115920269367052e-01 2.02907889905697916971272221076106e-03
            6.97517800276663990644010482355952e-01 1.18693569421395000995644863905909e-01 2.02907889905697916971272221076106e-03
            3.21121524868778346339937002085207e-02 1.39305721718615584114875360910446e-03 1.13321287603745677404105951513458e-04
            3.21121524868778346339937002085207e-02 9.66494790295935990442899310437497e-01 1.13321287603745677404105951513458e-04
            1.39305721718615584114875360910446e-03 3.21121524868778346339937002085207e-02 1.13321287603745677404105951513458e-04
            1.39305721718615584114875360910446e-03 9.66494790295935990442899310437497e-01 1.13321287603745677404105951513458e-04
            9.66494790295935990442899310437497e-01 3.21121524868778346339937002085207e-02 1.13321287603745677404105951513458e-04
            9.66494790295935990442899310437497e-01 1.39305721718615584114875360910446e-03 1.13321287603745677404105951513458e-04
            4.29640722006761482942849283972464e-02 1.02994751818414106286914133647770e-02 3.40926671570748306833165131024543e-04
            4.29640722006761482942849283972464e-02 9.46736452617482515670133125240682e-01 3.40926671570748306833165131024543e-04
            1.02994751818414106286914133647770e-02 4.29640722006761482942849283972464e-02 3.40926671570748306833165131024543e-04
            1.02994751818414106286914133647770e-02 9.46736452617482515670133125240682e-01 3.40926671570748306833165131024543e-04
            9.46736452617482515670133125240682e-01 4.29640722006761482942849283972464e-02 3.40926671570748306833165131024543e-04
            9.46736452617482515670133125240682e-01 1.02994751818414106286914133647770e-02 3.40926671570748306833165131024543e-04
            1.05248427300748837592836437693222e-02 2.58470999228818361717618934392249e-03 9.35465343078011979209562820614110e-05
            1.05248427300748837592836437693222e-02 9.86890447277636861933558520831866e-01 9.35465343078011979209562820614110e-05
            2.58470999228818361717618934392249e-03 1.05248427300748837592836437693222e-02 9.35465343078011979209562820614110e-05
            2.58470999228818361717618934392249e-03 9.86890447277636861933558520831866e-01 9.35465343078011979209562820614110e-05
            9.86890447277636861933558520831866e-01 1.05248427300748837592836437693222e-02 9.35465343078011979209562820614110e-05
            9.86890447277636861933558520831866e-01 2.58470999228818361717618934392249e-03 9.35465343078011979209562820614110e-05
            7.53790285629702960790154975256883e-02 3.94743127161576515260499320447707e-02 8.43314834911866161720750412200687e-04
            7.53790285629702960790154975256883e-02 8.85146658720872059333828474336769e-01 8.43314834911866161720750412200687e-04
            3.94743127161576515260499320447707e-02 7.53790285629702960790154975256883e-02 8.43314834911866161720750412200687e-04
            3.94743127161576515260499320447707e-02 8.85146658720872059333828474336769e-01 8.43314834911866161720750412200687e-04
            8.85146658720872059333828474336769e-01 7.53790285629702960790154975256883e-02 8.43314834911866161720750412200687e-04
            8.85146658720872059333828474336769e-01 3.94743127161576515260499320447707e-02 8.43314834911866161720750412200687e-04
            4.36333362089955989482348286401248e-01 1.22938385245525295258461895286928e-02 9.51274898228950094823674898947274e-04
            4.36333362089955989482348286401248e-01 5.51372799385491441093165576603496e-01 9.51274898228950094823674898947274e-04
            1.22938385245525295258461895286928e-02 4.36333362089955989482348286401248e-01 9.51274898228950094823674898947274e-04
            1.22938385245525295258461895286928e-02 5.51372799385491441093165576603496e-01 9.51274898228950094823674898947274e-04
            5.51372799385491441093165576603496e-01 4.36333362089955989482348286401248e-01 9.51274898228950094823674898947274e-04
            5.51372799385491441093165576603496e-01 1.22938385245525295258461895286928e-02 9.51274898228950094823674898947274e-04
            2.70490746514976954095743622019654e-01 2.58980924784568732402445512263967e-03 3.86683512649890390698581565942504e-04
            2.70490746514976954095743622019654e-01 7.26919444237177359013912791851908e-01 3.86683512649890390698581565942504e-04
            2.58980924784568732402445512263967e-03 2.70490746514976954095743622019654e-01 3.86683512649890390698581565942504e-04
            2.58980924784568732402445512263967e-03 7.26919444237177359013912791851908e-01 3.86683512649890390698581565942504e-04
            7.26919444237177359013912791851908e-01 2.70490746514976954095743622019654e-01 3.86683512649890390698581565942504e-04
            7.26919444237177359013912791851908e-01 2.58980924784568732402445512263967e-03 3.86683512649890390698581565942504e-04
            3.50813643381097384210676182192401e-01 3.75563888722968697297943663215847e-02 1.58619394545012016781160291145625e-03
            3.50813643381097384210676182192401e-01 6.11629967746605829326256298372755e-01 1.58619394545012016781160291145625e-03
            3.75563888722968697297943663215847e-02 3.50813643381097384210676182192401e-01 1.58619394545012016781160291145625e-03
            3.75563888722968697297943663215847e-02 6.11629967746605829326256298372755e-01 1.58619394545012016781160291145625e-03
            6.11629967746605829326256298372755e-01 3.50813643381097384210676182192401e-01 1.58619394545012016781160291145625e-03
            6.11629967746605829326256298372755e-01 3.75563888722968697297943663215847e-02 1.58619394545012016781160291145625e-03
            3.72765511974292140795483874171623e-01 1.70945467207073396875838255937197e-02 1.08426235493087126833533506697904e-03
            3.72765511974292140795483874171623e-01 6.10139941305000599314212195167784e-01 1.08426235493087126833533506697904e-03
            1.70945467207073396875838255937197e-02 3.72765511974292140795483874171623e-01 1.08426235493087126833533506697904e-03
            1.70945467207073396875838255937197e-02 6.10139941305000599314212195167784e-01 1.08426235493087126833533506697904e-03
            6.10139941305000599314212195167784e-01 3.72765511974292140795483874171623e-01 1.08426235493087126833533506697904e-03
            6.10139941305000599314212195167784e-01 1.70945467207073396875838255937197e-02 1.08426235493087126833533506697904e-03
            2.17107352263341546461106190690771e-01 8.57583162943158783253494448217680e-02 2.03705873682035734292816719914754e-03
            2.17107352263341546461106190690771e-01 6.97134331442342602969119980116375e-01 2.03705873682035734292816719914754e-03
            8.57583162943158783253494448217680e-02 2.17107352263341546461106190690771e-01 2.03705873682035734292816719914754e-03
            8.57583162943158783253494448217680e-02 6.97134331442342602969119980116375e-01 2.03705873682035734292816719914754e-03
            6.97134331442342602969119980116375e-01 2.17107352263341546461106190690771e-01 2.03705873682035734292816719914754e-03
            6.97134331442342602969119980116375e-01 8.57583162943158783253494448217680e-02 2.03705873682035734292816719914754e-03
            1.92756365009005836697042468586005e-01 5.79856449641143953388500165146979e-02 1.68604450719807568044583678812387e-03
            1.92756365009005836697042468586005e-01 7.49257990026879761025213610992068e-01 1.68604450719807568044583678812387e-03
            5.79856449641143953388500165146979e-02 1.92756365009005836697042468586005e-01 1.68604450719807568044583678812387e-03
            5.79856449641143953388500165146979e-02 7.49257990026879761025213610992068e-01 1.68604450719807568044583678812387e-03
            7.49257990026879761025213610992068e-01 1.92756365009005836697042468586005e-01 1.68604450719807568044583678812387e-03
            7.49257990026879761025213610992068e-01 5.79856449641143953388500165146979e-02 1.68604450719807568044583678812387e-03
            4.20685030686416272516225944855250e-01 3.38481728829903155686409377267410e-02 1.66873270767465111188987414436724e-03
            4.20685030686416272516225944855250e-01 5.45466796430593370281769693974638e-01 1.66873270767465111188987414436724e-03
            3.38481728829903155686409377267410e-02 4.20685030686416272516225944855250e-01 1.66873270767465111188987414436724e-03
            3.38481728829903155686409377267410e-02 5.45466796430593370281769693974638e-01 1.66873270767465111188987414436724e-03
            5.45466796430593370281769693974638e-01 4.20685030686416272516225944855250e-01 1.66873270767465111188987414436724e-03
            5.45466796430593370281769693974638e-01 3.38481728829903155686409377267410e-02 1.66873270767465111188987414436724e-03
            2.12485185437091050886238008388318e-01 2.52669138520864814539956810790500e-03 3.72150230356113652721572471193667e-04
            2.12485185437091050886238008388318e-01 7.84988123177700281019042449770495e-01 3.72150230356113652721572471193667e-04
            2.52669138520864814539956810790500e-03 2.12485185437091050886238008388318e-01 3.72150230356113652721572471193667e-04
            2.52669138520864814539956810790500e-03 7.84988123177700281019042449770495e-01 3.72150230356113652721572471193667e-04
            7.84988123177700281019042449770495e-01 2.12485185437091050886238008388318e-01 3.72150230356113652721572471193667e-04
            7.84988123177700281019042449770495e-01 2.52669138520864814539956810790500e-03 3.72150230356113652721572471193667e-04
            3.19257208729187336171406741414103e-01 1.06565421155276375542175770760878e-02 8.74502828754643128650325678563604e-04
            3.19257208729187336171406741414103e-01 6.70086249155284918721520170947770e-01 8.74502828754643128650325678563604e-04
            1.06565421155276375542175770760878e-02 3.19257208729187336171406741414103e-01 8.74502828754643128650325678563604e-04
            1.06565421155276375542175770760878e-02 6.70086249155284918721520170947770e-01 8.74502828754643128650325678563604e-04
            6.70086249155284918721520170947770e-01 3.19257208729187336171406741414103e-01 8.74502828754643128650325678563604e-04
            6.70086249155284918721520170947770e-01 1.06565421155276375542175770760878e-02 8.74502828754643128650325678563604e-04
            2.51638235512572583374435453151818e-01 1.34296732422585305549711520711753e-02 9.51584197719013689421851953653686e-04
            2.51638235512572583374435453151818e-01 7.34932091245168894744210774661042e-01 9.51584197719013689421851953653686e-04
            1.34296732422585305549711520711753e-02 2.51638235512572583374435453151818e-01 9.51584197719013689421851953653686e-04
            1.34296732422585305549711520711753e-02 7.34932091245168894744210774661042e-01 9.51584197719013689421851953653686e-04
            7.34932091245168894744210774661042e-01 2.51638235512572583374435453151818e-01 9.51584197719013689421851953653686e-04
            7.34932091245168894744210774661042e-01 1.34296732422585305549711520711753e-02 9.51584197719013689421851953653686e-04
            1.52211981456809364310345245030476e-01 8.83417782932650746952063514072506e-02 1.97907224537829679739120436465782e-03
            1.52211981456809364310345245030476e-01 7.59446240249925574872236211376730e-01 1.97907224537829679739120436465782e-03
            8.83417782932650746952063514072506e-02 1.52211981456809364310345245030476e-01 1.97907224537829679739120436465782e-03
            8.83417782932650746952063514072506e-02 7.59446240249925574872236211376730e-01 1.97907224537829679739120436465782e-03
            7.59446240249925574872236211376730e-01 1.52211981456809364310345245030476e-01 1.97907224537829679739120436465782e-03
            7.59446240249925574872236211376730e-01 8.83417782932650746952063514072506e-02 1.97907224537829679739120436465782e-03
            1.11142094794033280402700825106876e-01 3.29221182111608043352468655484699e-02 1.08166500595875464049711034419943e-03
            1.11142094794033280402700825106876e-01 8.55935786994805991589885252324166e-01 1.08166500595875464049711034419943e-03
            3.29221182111608043352468655484699e-02 1.11142094794033280402700825106876e-01 1.08166500595875464049711034419943e-03
            3.29221182111608043352468655484699e-02 8.55935786994805991589885252324166e-01 1.08166500595875464049711034419943e-03
            8.55935786994805991589885252324166e-01 1.11142094794033280402700825106876e-01 1.08166500595875464049711034419943e-03
            8.55935786994805991589885252324166e-01 3.29221182111608043352468655484699e-02 1.08166500595875464049711034419943e-03
            3.58150058851250285840706055751070e-01 1.94800760777327719441842646119767e-01 3.13337069827091148355169281103372e-03
            3.58150058851250285840706055751070e-01 4.47049180371421994717451298129163e-01 3.13337069827091148355169281103372e-03
            1.94800760777327719441842646119767e-01 3.58150058851250285840706055751070e-01 3.13337069827091148355169281103372e-03
            1.94800760777327719441842646119767e-01 4.47049180371421994717451298129163e-01 3.13337069827091148355169281103372e-03
            4.47049180371421994717451298129163e-01 3.58150058851250285840706055751070e-01 3.13337069827091148355169281103372e-03
            4.47049180371421994717451298129163e-01 1.94800760777327719441842646119767e-01 3.13337069827091148355169281103372e-03
            1.63799856956150069464683838305064e-01 3.14610159618917972768059598820400e-02 1.21573547237640460817742482646509e-03
            1.63799856956150069464683838305064e-01 8.04739127081958161014085817441810e-01 1.21573547237640460817742482646509e-03
            3.14610159618917972768059598820400e-02 1.63799856956150069464683838305064e-01 1.21573547237640460817742482646509e-03
            3.14610159618917972768059598820400e-02 8.04739127081958161014085817441810e-01 1.21573547237640460817742482646509e-03
            8.04739127081958161014085817441810e-01 1.63799856956150069464683838305064e-01 1.21573547237640460817742482646509e-03
            8.04739127081958161014085817441810e-01 3.14610159618917972768059598820400e-02 1.21573547237640460817742482646509e-03
            1.02112773336674317326000505090633e-01 6.73683101645258797240956027962966e-02 1.41328904880122746372250741586640e-03
            1.02112773336674317326000505090633e-01 8.30518916498799830705479507741984e-01 1.41328904880122746372250741586640e-03
            6.73683101645258797240956027962966e-02 1.02112773336674317326000505090633e-01 1.41328904880122746372250741586640e-03
            6.73683101645258797240956027962966e-02 8.30518916498799830705479507741984e-01 1.41328904880122746372250741586640e-03
            8.30518916498799830705479507741984e-01 1.02112773336674317326000505090633e-01 1.41328904880122746372250741586640e-03
            8.30518916498799830705479507741984e-01 6.73683101645258797240956027962966e-02 1.41328904880122746372250741586640e-03
            2.33555703019351579285967090982012e-01 1.86565244199972779526319754950237e-01 3.23767811274164188720114943009776e-03
            2.33555703019351579285967090982012e-01 5.79879052780675641187713154067751e-01 3.23767811274164188720114943009776e-03
            1.86565244199972779526319754950237e-01 2.33555703019351579285967090982012e-01 3.23767811274164188720114943009776e-03
            1.86565244199972779526319754950237e-01 5.79879052780675641187713154067751e-01 3.23767811274164188720114943009776e-03
            5.79879052780675641187713154067751e-01 2.33555703019351579285967090982012e-01 3.23767811274164188720114943009776e-03
            5.79879052780675641187713154067751e-01 1.86565244199972779526319754950237e-01 3.23767811274164188720114943009776e-03
            3.21416462831711324810157748288475e-01 6.38834834123953448914434716243704e-02 2.12424122562709363423727459974089e-03
            3.21416462831711324810157748288475e-01 6.14700053755893316420610972272698e-01 2.12424122562709363423727459974089e-03
            6.38834834123953448914434716243704e-02 3.21416462831711324810157748288475e-01 2.12424122562709363423727459974089e-03
            6.38834834123953448914434716243704e-02 6.14700053755893316420610972272698e-01 2.12424122562709363423727459974089e-03
            6.14700053755893316420610972272698e-01 3.21416462831711324810157748288475e-01 2.12424122562709363423727459974089e-03
            6.14700053755893316420610972272698e-01 6.38834834123953448914434716243704e-02 2.12424122562709363423727459974089e-03
            2.23336184895712552833657582596061e-01 3.25712217949498672453501058043912e-02 1.45504295453112724063882144065474e-03
            2.23336184895712552833657582596061e-01 7.44092593309337635432143542857375e-01 1.45504295453112724063882144065474e-03
            3.25712217949498672453501058043912e-02 2.23336184895712552833657582596061e-01 1.45504295453112724063882144065474e-03
            3.25712217949498672453501058043912e-02 7.44092593309337635432143542857375e-01 1.45504295453112724063882144065474e-03
            7.44092593309337635432143542857375e-01 2.23336184895712552833657582596061e-01 1.45504295453112724063882144065474e-03
            7.44092593309337635432143542857375e-01 3.25712217949498672453501058043912e-02 1.45504295453112724063882144065474e-03
            1.90588118946320272950600838157698e-01 1.32734827561577940074899473188452e-02 8.77197165465703880028736083573904e-04
            1.90588118946320272950600838157698e-01 7.96138398297521865387693651427981e-01 8.77197165465703880028736083573904e-04
            1.32734827561577940074899473188452e-02 1.90588118946320272950600838157698e-01 8.77197165465703880028736083573904e-04
            1.32734827561577940074899473188452e-02 7.96138398297521865387693651427981e-01 8.77197165465703880028736083573904e-04
            7.96138398297521865387693651427981e-01 1.90588118946320272950600838157698e-01 8.77197165465703880028736083573904e-04
            7.96138398297521865387693651427981e-01 1.32734827561577940074899473188452e-02 8.77197165465703880028736083573904e-04
            2.27355840245479501326997251453577e-01 1.36233775509128074432396715565119e-01 2.66390718176628960023322179040406e-03
            2.27355840245479501326997251453577e-01 6.36410384245392424240606032981304e-01 2.66390718176628960023322179040406e-03
            1.36233775509128074432396715565119e-01 2.27355840245479501326997251453577e-01 2.66390718176628960023322179040406e-03
            1.36233775509128074432396715565119e-01 6.36410384245392424240606032981304e-01 2.66390718176628960023322179040406e-03
            6.36410384245392424240606032981304e-01 2.27355840245479501326997251453577e-01 2.66390718176628960023322179040406e-03
            6.36410384245392424240606032981304e-01 1.36233775509128074432396715565119e-01 2.66390718176628960023322179040406e-03
            1.56767922787160035591469409155252e-01 2.61537753388922135366900789676947e-03 3.55867757526690084608766984075601e-04
            1.56767922787160035591469409155252e-01 8.40616699678950807239630194089841e-01 3.55867757526690084608766984075601e-04
            2.61537753388922135366900789676947e-03 1.56767922787160035591469409155252e-01 3.55867757526690084608766984075601e-04
            2.61537753388922135366900789676947e-03 8.40616699678950807239630194089841e-01 3.55867757526690084608766984075601e-04
            8.40616699678950807239630194089841e-01 1.56767922787160035591469409155252e-01 3.55867757526690084608766984075601e-04
            8.40616699678950807239630194089841e-01 2.61537753388922135366900789676947e-03 3.55867757526690084608766984075601e-04
            1.34033334986778468467250036155747e-01 1.34812112051568063625373028457943e-02 7.84259809071718733824996050429945e-04
            1.34033334986778468467250036155747e-01 8.52485453808064730374383088928880e-01 7.84259809071718733824996050429945e-04
            1.34812112051568063625373028457943e-02 1.34033334986778468467250036155747e-01 7.84259809071718733824996050429945e-04
            1.34812112051568063625373028457943e-02 8.52485453808064730374383088928880e-01 7.84259809071718733824996050429945e-04
            8.52485453808064730374383088928880e-01 1.34033334986778468467250036155747e-01 7.84259809071718733824996050429945e-04
            8.52485453808064730374383088928880e-01 1.34812112051568063625373028457943e-02 7.84259809071718733824996050429945e-04
            2.95721175745453590533173837684444e-01 1.91884104352206263932600904809078e-01 3.45484185169266750856098546762496e-03
            2.95721175745453590533173837684444e-01 5.12394719902340090023074026248651e-01 3.45484185169266750856098546762496e-03
            1.91884104352206263932600904809078e-01 2.95721175745453590533173837684444e-01 3.45484185169266750856098546762496e-03
            1.91884104352206263932600904809078e-01 5.12394719902340090023074026248651e-01 3.45484185169266750856098546762496e-03
            5.12394719902340090023074026248651e-01 2.95721175745453590533173837684444e-01 3.45484185169266750856098546762496e-03
            5.12394719902340090023074026248651e-01 1.91884104352206263932600904809078e-01 3.45484185169266750856098546762496e-03
            1.06719271813000365911605626934033e-01 2.54728163216983664179915436420742e-03 3.03883444669208048309616554405466e-04
            1.06719271813000365911605626934033e-01 8.90733446554829777497275244968478e-01 3.03883444669208048309616554405466e-04
            2.54728163216983664179915436420742e-03 1.06719271813000365911605626934033e-01 3.03883444669208048309616554405466e-04
            2.54728163216983664179915436420742e-03 8.90733446554829777497275244968478e-01 3.03883444669208048309616554405466e-04
            8.90733446554829777497275244968478e-01 1.06719271813000365911605626934033e-01 3.03883444669208048309616554405466e-04
            8.90733446554829777497275244968478e-01 2.54728163216983664179915436420742e-03 3.03883444669208048309616554405466e-04
            4.12566419483916535515533041689196e-01 1.01037294355759685315909734981688e-01 2.92995302527590106150823245911852e-03
            4.12566419483916535515533041689196e-01 4.86396286160323820801920646772487e-01 2.92995302527590106150823245911852e-03
            1.01037294355759685315909734981688e-01 4.12566419483916535515533041689196e-01 2.92995302527590106150823245911852e-03
            1.01037294355759685315909734981688e-01 4.86396286160323820801920646772487e-01 2.92995302527590106150823245911852e-03
            4.86396286160323820801920646772487e-01 4.12566419483916535515533041689196e-01 2.92995302527590106150823245911852e-03
            4.86396286160323820801920646772487e-01 1.01037294355759685315909734981688e-01 2.92995302527590106150823245911852e-03
            3.07851982992886075152227931539528e-01 2.49750741165288780987197014837875e-01 3.88259302031484414621798073596892e-03
            3.07851982992886075152227931539528e-01 4.42397275841825143860575053622597e-01 3.88259302031484414621798073596892e-03
            2.49750741165288780987197014837875e-01 3.07851982992886075152227931539528e-01 3.88259302031484414621798073596892e-03
            2.49750741165288780987197014837875e-01 4.42397275841825143860575053622597e-01 3.88259302031484414621798073596892e-03
            4.42397275841825143860575053622597e-01 3.07851982992886075152227931539528e-01 3.88259302031484414621798073596892e-03
            4.42397275841825143860575053622597e-01 2.49750741165288780987197014837875e-01 3.88259302031484414621798073596892e-03
            8.70205486996966370849548866317491e-02 1.37126252885580324908376681491973e-02 6.55020416493056270579309341428598e-04
            8.70205486996966370849548866317491e-02 8.99266826011745346036718729010317e-01 6.55020416493056270579309341428598e-04
            1.37126252885580324908376681491973e-02 8.70205486996966370849548866317491e-02 6.55020416493056270579309341428598e-04
            1.37126252885580324908376681491973e-02 8.99266826011745346036718729010317e-01 6.55020416493056270579309341428598e-04
            8.99266826011745346036718729010317e-01 8.70205486996966370849548866317491e-02 6.55020416493056270579309341428598e-04
            8.99266826011745346036718729010317e-01 1.37126252885580324908376681491973e-02 6.55020416493056270579309341428598e-04
            3.89428331681219708215735408884939e-01 3.34459543340264765787228640192552e-03 5.28548624586135194075575682859380e-04
            3.89428331681219708215735408884939e-01 6.07227072885377694433373108040541e-01 5.28548624586135194075575682859380e-04
            3.34459543340264765787228640192552e-03 3.89428331681219708215735408884939e-01 5.28548624586135194075575682859380e-04
            3.34459543340264765787228640192552e-03 6.07227072885377694433373108040541e-01 5.28548624586135194075575682859380e-04
            6.07227072885377694433373108040541e-01 3.89428331681219708215735408884939e-01 5.28548624586135194075575682859380e-04
            6.07227072885377694433373108040541e-01 3.34459543340264765787228640192552e-03 5.28548624586135194075575682859380e-04
            2.61293970314177004254929670423735e-01 5.56746304306052139376781440205377e-02 2.02059882537530084364862936752161e-03
            2.61293970314177004254929670423735e-01 6.83031399255217830379649512906326e-01 2.02059882537530084364862936752161e-03
            5.56746304306052139376781440205377e-02 2.61293970314177004254929670423735e-01 2.02059882537530084364862936752161e-03
            5.56746304306052139376781440205377e-02 6.83031399255217830379649512906326e-01 2.02059882537530084364862936752161e-03
            6.83031399255217830379649512906326e-01 2.61293970314177004254929670423735e-01 2.02059882537530084364862936752161e-03
            6.83031399255217830379649512906326e-01 5.56746304306052139376781440205377e-02 2.02059882537530084364862936752161e-03
            6.49755331098118837918065082703833e-02 2.80375859354026635977819026379620e-03 2.62123945672969142772767980886783e-04
            6.49755331098118837918065082703833e-02 9.32220708296647826429648375778925e-01 2.62123945672969142772767980886783e-04
            2.80375859354026635977819026379620e-03 6.49755331098118837918065082703833e-02 2.62123945672969142772767980886783e-04
            2.80375859354026635977819026379620e-03 9.32220708296647826429648375778925e-01 2.62123945672969142772767980886783e-04
            9.32220708296647826429648375778925e-01 6.49755331098118837918065082703833e-02 2.62123945672969142772767980886783e-04
            9.32220708296647826429648375778925e-01 2.80375859354026635977819026379620e-03 2.62123945672969142772767980886783e-04
            2.71950006098130969700576997638564e-01 9.63505000772398451092115578830999e-02 2.60243400316005494068982706323823e-03
            2.71950006098130969700576997638564e-01 6.31699493824629088045696789777139e-01 2.60243400316005494068982706323823e-03
            9.63505000772398451092115578830999e-02 2.71950006098130969700576997638564e-01 2.60243400316005494068982706323823e-03
            9.63505000772398451092115578830999e-02 6.31699493824629088045696789777139e-01 2.60243400316005494068982706323823e-03
            6.31699493824629088045696789777139e-01 2.71950006098130969700576997638564e-01 2.60243400316005494068982706323823e-03
            6.31699493824629088045696789777139e-01 9.63505000772398451092115578830999e-02 2.60243400316005494068982706323823e-03
            2.93090364272546199941871236660518e-01 2.91991066004359950403213019853865e-02 1.51665566313493650231480902590420e-03
            2.93090364272546199941871236660518e-01 6.77710529127017791140019653539639e-01 1.51665566313493650231480902590420e-03
            2.91991066004359950403213019853865e-02 2.93090364272546199941871236660518e-01 1.51665566313493650231480902590420e-03
            2.91991066004359950403213019853865e-02 6.77710529127017791140019653539639e-01 1.51665566313493650231480902590420e-03
            6.77710529127017791140019653539639e-01 2.93090364272546199941871236660518e-01 1.51665566313493650231480902590420e-03
            6.77710529127017791140019653539639e-01 2.91991066004359950403213019853865e-02 1.51665566313493650231480902590420e-03
            3.94664826587819206338281219359487e-01 6.35745506621691741644397666277655e-02 2.43427618019893474143877654114476e-03
            3.94664826587819206338281219359487e-01 5.41760622750011577863915590569377e-01 2.43427618019893474143877654114476e-03
            6.35745506621691741644397666277655e-02 3.94664826587819206338281219359487e-01 2.43427618019893474143877654114476e-03
            6.35745506621691741644397666277655e-02 5.41760622750011577863915590569377e-01 2.43427618019893474143877654114476e-03
            5.41760622750011577863915590569377e-01 3.94664826587819206338281219359487e-01 2.43427618019893474143877654114476e-03
            5.41760622750011577863915590569377e-01 6.35745506621691741644397666277655e-02 2.43427618019893474143877654114476e-03
            3.56852218192982795219592162538902e-01 1.43928580065543848531817161529034e-01 3.30143630904448318746924329047943e-03
            3.56852218192982795219592162538902e-01 4.99219201741473384004166291560978e-01 3.30143630904448318746924329047943e-03
            1.43928580065543848531817161529034e-01 3.56852218192982795219592162538902e-01 3.30143630904448318746924329047943e-03
            1.43928580065543848531817161529034e-01 4.99219201741473384004166291560978e-01 3.30143630904448318746924329047943e-03
            4.99219201741473384004166291560978e-01 3.56852218192982795219592162538902e-01 3.30143630904448318746924329047943e-03
            4.99219201741473384004166291560978e-01 1.43928580065543848531817161529034e-01 3.30143630904448318746924329047943e-03
            3.41361351638876264757982426090166e-01 9.92086997532997810766985935515549e-02 2.82656834415620202352603662632191e-03
            3.41361351638876264757982426090166e-01 5.59429948607823912531955556914909e-01 2.82656834415620202352603662632191e-03
            9.92086997532997810766985935515549e-02 3.41361351638876264757982426090166e-01 2.82656834415620202352603662632191e-03
            9.92086997532997810766985935515549e-02 5.59429948607823912531955556914909e-01 2.82656834415620202352603662632191e-03
            5.59429948607823912531955556914909e-01 3.41361351638876264757982426090166e-01 2.82656834415620202352603662632191e-03
            5.59429948607823912531955556914909e-01 9.92086997532997810766985935515549e-02 2.82656834415620202352603662632191e-03
            2.89512941079514585229759404683136e-01 1.40388921421462892213938289387443e-01 3.09596450617196901691774435505522e-03
            2.89512941079514585229759404683136e-01 5.70098137499022605823029152816162e-01 3.09596450617196901691774435505522e-03
            1.40388921421462892213938289387443e-01 2.89512941079514585229759404683136e-01 3.09596450617196901691774435505522e-03
            1.40388921421462892213938289387443e-01 5.70098137499022605823029152816162e-01 3.09596450617196901691774435505522e-03
            5.70098137499022605823029152816162e-01 2.89512941079514585229759404683136e-01 3.09596450617196901691774435505522e-03
            5.70098137499022605823029152816162e-01 1.40388921421462892213938289387443e-01 3.09596450617196901691774435505522e-03
            ];
        
    case  43
        
        % ALG. DEG.:   43
        % PTS CARD.:  339
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.5e-15
        
        xyw=[
            4.11780170859933791760454369068611e-01 4.11780170859933791760454369068611e-01 1.53998278459081686456166870868856e-03
            4.11780170859933791760454369068611e-01 1.76439658280132416479091261862777e-01 1.53998278459081686456166870868856e-03
            1.76439658280132416479091261862777e-01 4.11780170859933791760454369068611e-01 1.53998278459081686456166870868856e-03
            4.75575867099131466275707680324558e-01 4.75575867099131466275707680324558e-01 1.24367411012324207385260077529665e-03
            4.75575867099131466275707680324558e-01 4.88482658017370674485846393508837e-02 1.24367411012324207385260077529665e-03
            4.88482658017370674485846393508837e-02 4.75575867099131466275707680324558e-01 1.24367411012324207385260077529665e-03
            4.99179836734477966331269271904603e-01 4.99179836734477966331269271904603e-01 3.04307861090390098426222698080323e-04
            4.99179836734477966331269271904603e-01 1.64032653104406733746145619079471e-03 3.04307861090390098426222698080323e-04
            1.64032653104406733746145619079471e-03 4.99179836734477966331269271904603e-01 3.04307861090390098426222698080323e-04
            2.96247988168869824665563328380813e-01 2.96247988168869824665563328380813e-01 2.93904116687957860792890230072771e-03
            2.96247988168869824665563328380813e-01 4.07504023662260350668873343238374e-01 2.93904116687957860792890230072771e-03
            4.07504023662260350668873343238374e-01 2.96247988168869824665563328380813e-01 2.93904116687957860792890230072771e-03
            3.53188804887000662446894239110406e-01 3.53188804887000662446894239110406e-01 3.43424448325747823027032090692501e-03
            3.53188804887000662446894239110406e-01 2.93622390225998675106211521779187e-01 3.43424448325747823027032090692501e-03
            2.93622390225998675106211521779187e-01 3.53188804887000662446894239110406e-01 3.43424448325747823027032090692501e-03
            4.95105980802769907178628727706382e-01 4.95105980802769907178628727706382e-01 8.07153621957877142993609265175792e-04
            4.95105980802769907178628727706382e-01 9.78803839446018564274254458723590e-03 8.07153621957877142993609265175792e-04
            9.78803839446018564274254458723590e-03 4.95105980802769907178628727706382e-01 8.07153621957877142993609265175792e-04
            3.29098159605345705336354455994297e-02 3.29098159605345705336354455994297e-02 5.41811214635524313054659728550178e-04
            3.29098159605345705336354455994297e-02 9.34180368078930900566092532244511e-01 5.41811214635524313054659728550178e-04
            9.34180368078930900566092532244511e-01 3.29098159605345705336354455994297e-02 5.41811214635524313054659728550178e-04
            4.86563751130688337820373590147938e-01 4.86563751130688337820373590147938e-01 1.35830253414758882375690252075628e-03
            4.86563751130688337820373590147938e-01 2.68724977386233243592528197041247e-02 1.35830253414758882375690252075628e-03
            2.68724977386233243592528197041247e-02 4.86563751130688337820373590147938e-01 1.35830253414758882375690252075628e-03
            6.27150483897616628503612901113229e-02 6.27150483897616628503612901113229e-02 1.01216465227482710874384874699672e-03
            6.27150483897616628503612901113229e-02 8.74569903220476674299277419777354e-01 1.01216465227482710874384874699672e-03
            8.74569903220476674299277419777354e-01 6.27150483897616628503612901113229e-02 1.01216465227482710874384874699672e-03
            1.44743559770221352778207801748067e-01 1.44743559770221352778207801748067e-01 2.12271596053216233396243595166197e-03
            1.44743559770221352778207801748067e-01 7.10512880459557294443584396503866e-01 2.12271596053216233396243595166197e-03
            7.10512880459557294443584396503866e-01 1.44743559770221352778207801748067e-01 2.12271596053216233396243595166197e-03
            1.33646206082615737198437955157715e-02 1.33646206082615737198437955157715e-02 2.36051052789603533306125870971925e-04
            1.33646206082615737198437955157715e-02 9.73270758783476797049161177710630e-01 2.36051052789603533306125870971925e-04
            9.73270758783476797049161177710630e-01 1.33646206082615737198437955157715e-02 2.36051052789603533306125870971925e-04
            2.56494137002013589468618626199259e-03 2.56494137002013589468618626199259e-03 4.28662883399411086867564213420678e-05
            2.56494137002013589468618626199259e-03 9.94870117259959774180799740861403e-01 4.28662883399411086867564213420678e-05
            9.94870117259959774180799740861403e-01 2.56494137002013589468618626199259e-03 4.28662883399411086867564213420678e-05
            1.87772242778578402067068964242935e-01 1.87772242778578402067068964242935e-01 2.66763767119809497990745583706484e-03
            1.87772242778578402067068964242935e-01 6.24455514442843195865862071514130e-01 2.66763767119809497990745583706484e-03
            6.24455514442843195865862071514130e-01 1.87772242778578402067068964242935e-01 2.66763767119809497990745583706484e-03
            3.82564308786935480810598164680414e-01 3.82564308786935480810598164680414e-01 3.47484043452829039280715939241873e-03
            3.82564308786935480810598164680414e-01 2.34871382426129038378803670639172e-01 3.47484043452829039280715939241873e-03
            2.34871382426129038378803670639172e-01 3.82564308786935480810598164680414e-01 3.47484043452829039280715939241873e-03
            4.52892260845951022396604912501061e-01 4.52892260845951022396604912501061e-01 2.66640271020050691891389860188610e-03
            4.52892260845951022396604912501061e-01 9.42154783080979552067901749978773e-02 2.66640271020050691891389860188610e-03
            9.42154783080979552067901749978773e-02 4.52892260845951022396604912501061e-01 2.66640271020050691891389860188610e-03
            9.28670462140478714108127178405994e-02 9.28670462140478714108127178405994e-02 1.64535307292699864781249097234195e-03
            9.28670462140478714108127178405994e-02 8.14265907571904312689525795576628e-01 1.64535307292699864781249097234195e-03
            8.14265907571904312689525795576628e-01 9.28670462140478714108127178405994e-02 1.64535307292699864781249097234195e-03
            2.52484932649090720868656489983550e-01 2.52484932649090720868656489983550e-01 3.65710457156017071725373490664879e-03
            2.52484932649090720868656489983550e-01 4.95030134701818558262687020032899e-01 3.65710457156017071725373490664879e-03
            4.95030134701818558262687020032899e-01 2.52484932649090720868656489983550e-01 3.65710457156017071725373490664879e-03
            2.29739436527249235586722875268606e-01 1.45453795387832596208710356222582e-01 1.95631917040878493957434258732064e-03
            2.29739436527249235586722875268606e-01 6.24806768084918084937839921622071e-01 1.95631917040878493957434258732064e-03
            1.45453795387832596208710356222582e-01 2.29739436527249235586722875268606e-01 1.95631917040878493957434258732064e-03
            1.45453795387832596208710356222582e-01 6.24806768084918084937839921622071e-01 1.95631917040878493957434258732064e-03
            6.24806768084918084937839921622071e-01 2.29739436527249235586722875268606e-01 1.95631917040878493957434258732064e-03
            6.24806768084918084937839921622071e-01 1.45453795387832596208710356222582e-01 1.95631917040878493957434258732064e-03
            3.60558076088783818757832477785996e-01 3.24154748417785493197484925076424e-02 1.32877988603856043124673558253335e-03
            3.60558076088783818757832477785996e-01 6.07026449069437590289055606262991e-01 1.32877988603856043124673558253335e-03
            3.24154748417785493197484925076424e-02 3.60558076088783818757832477785996e-01 1.32877988603856043124673558253335e-03
            3.24154748417785493197484925076424e-02 6.07026449069437590289055606262991e-01 1.32877988603856043124673558253335e-03
            6.07026449069437590289055606262991e-01 3.60558076088783818757832477785996e-01 1.32877988603856043124673558253335e-03
            6.07026449069437590289055606262991e-01 3.24154748417785493197484925076424e-02 1.32877988603856043124673558253335e-03
            9.59172276669436746354691081251076e-02 1.24129082244383796146403042826023e-02 5.20192928808181458133486074757457e-04
            9.59172276669436746354691081251076e-02 8.91669864108617926667932351847412e-01 5.20192928808181458133486074757457e-04
            1.24129082244383796146403042826023e-02 9.59172276669436746354691081251076e-02 5.20192928808181458133486074757457e-04
            1.24129082244383796146403042826023e-02 8.91669864108617926667932351847412e-01 5.20192928808181458133486074757457e-04
            8.91669864108617926667932351847412e-01 9.59172276669436746354691081251076e-02 5.20192928808181458133486074757457e-04
            8.91669864108617926667932351847412e-01 1.24129082244383796146403042826023e-02 5.20192928808181458133486074757457e-04
            1.89801510111088478360841236280976e-01 1.37858117891493076045961174713739e-01 1.93441112276208177986336345099971e-03
            1.89801510111088478360841236280976e-01 6.72340371997418473348773204634199e-01 1.93441112276208177986336345099971e-03
            1.37858117891493076045961174713739e-01 1.89801510111088478360841236280976e-01 1.93441112276208177986336345099971e-03
            1.37858117891493076045961174713739e-01 6.72340371997418473348773204634199e-01 1.93441112276208177986336345099971e-03
            6.72340371997418473348773204634199e-01 1.89801510111088478360841236280976e-01 1.93441112276208177986336345099971e-03
            6.72340371997418473348773204634199e-01 1.37858117891493076045961174713739e-01 1.93441112276208177986336345099971e-03
            4.01058451292605144811176387520391e-01 1.35336131153596217480128416355001e-01 2.71800396079253025680144695286344e-03
            4.01058451292605144811176387520391e-01 4.63605417553798582197543964866782e-01 2.71800396079253025680144695286344e-03
            1.35336131153596217480128416355001e-01 4.01058451292605144811176387520391e-01 2.71800396079253025680144695286344e-03
            1.35336131153596217480128416355001e-01 4.63605417553798582197543964866782e-01 2.71800396079253025680144695286344e-03
            4.63605417553798582197543964866782e-01 4.01058451292605144811176387520391e-01 2.71800396079253025680144695286344e-03
            4.63605417553798582197543964866782e-01 1.35336131153596217480128416355001e-01 2.71800396079253025680144695286344e-03
            2.77582532276102134094486473259167e-01 1.43526791892476662804156717356818e-01 2.56651053850672590181392074271116e-03
            2.77582532276102134094486473259167e-01 5.78890675831421286368083656270755e-01 2.56651053850672590181392074271116e-03
            1.43526791892476662804156717356818e-01 2.77582532276102134094486473259167e-01 2.56651053850672590181392074271116e-03
            1.43526791892476662804156717356818e-01 5.78890675831421286368083656270755e-01 2.56651053850672590181392074271116e-03
            5.78890675831421286368083656270755e-01 2.77582532276102134094486473259167e-01 2.56651053850672590181392074271116e-03
            5.78890675831421286368083656270755e-01 1.43526791892476662804156717356818e-01 2.56651053850672590181392074271116e-03
            3.06118376690335036727219630847685e-01 3.54515915965141689691897397551656e-02 1.36244263020324067083033359182309e-03
            3.06118376690335036727219630847685e-01 6.58430031713150842875847956747748e-01 1.36244263020324067083033359182309e-03
            3.54515915965141689691897397551656e-02 3.06118376690335036727219630847685e-01 1.36244263020324067083033359182309e-03
            3.54515915965141689691897397551656e-02 6.58430031713150842875847956747748e-01 1.36244263020324067083033359182309e-03
            6.58430031713150842875847956747748e-01 3.06118376690335036727219630847685e-01 1.36244263020324067083033359182309e-03
            6.58430031713150842875847956747748e-01 3.54515915965141689691897397551656e-02 1.36244263020324067083033359182309e-03
            4.18289170173424762477765170842758e-01 3.54185331313221704596649885843362e-02 1.54096391385130193456276526120519e-03
            4.18289170173424762477765170842758e-01 5.46292296695253032368100321036763e-01 1.54096391385130193456276526120519e-03
            3.54185331313221704596649885843362e-02 4.18289170173424762477765170842758e-01 1.54096391385130193456276526120519e-03
            3.54185331313221704596649885843362e-02 5.46292296695253032368100321036763e-01 1.54096391385130193456276526120519e-03
            5.46292296695253032368100321036763e-01 4.18289170173424762477765170842758e-01 1.54096391385130193456276526120519e-03
            5.46292296695253032368100321036763e-01 3.54185331313221704596649885843362e-02 1.54096391385130193456276526120519e-03
            9.65099243357523811637221911041706e-02 5.57938681257981461403261391751585e-02 1.09800733044825558383517094540593e-03
            9.65099243357523811637221911041706e-02 8.47696207538449542084890708792955e-01 1.09800733044825558383517094540593e-03
            5.57938681257981461403261391751585e-02 9.65099243357523811637221911041706e-02 1.09800733044825558383517094540593e-03
            5.57938681257981461403261391751585e-02 8.47696207538449542084890708792955e-01 1.09800733044825558383517094540593e-03
            8.47696207538449542084890708792955e-01 9.65099243357523811637221911041706e-02 1.09800733044825558383517094540593e-03
            8.47696207538449542084890708792955e-01 5.57938681257981461403261391751585e-02 1.09800733044825558383517094540593e-03
            1.32839741110529208079071850079345e-01 6.39762347898086281006158060336020e-02 1.35358581459473457930808848459492e-03
            1.32839741110529208079071850079345e-01 8.03184024099662163820312343887053e-01 1.35358581459473457930808848459492e-03
            6.39762347898086281006158060336020e-02 1.32839741110529208079071850079345e-01 1.35358581459473457930808848459492e-03
            6.39762347898086281006158060336020e-02 8.03184024099662163820312343887053e-01 1.35358581459473457930808848459492e-03
            8.03184024099662163820312343887053e-01 1.32839741110529208079071850079345e-01 1.35358581459473457930808848459492e-03
            8.03184024099662163820312343887053e-01 6.39762347898086281006158060336020e-02 1.35358581459473457930808848459492e-03
            9.69643297980663232404907603267930e-02 3.02625797417711110692817300105162e-02 8.86625283151499577784238059763311e-04
            9.69643297980663232404907603267930e-02 8.72773090460162492831841518636793e-01 8.86625283151499577784238059763311e-04
            3.02625797417711110692817300105162e-02 9.69643297980663232404907603267930e-02 8.86625283151499577784238059763311e-04
            3.02625797417711110692817300105162e-02 8.72773090460162492831841518636793e-01 8.86625283151499577784238059763311e-04
            8.72773090460162492831841518636793e-01 9.69643297980663232404907603267930e-02 8.86625283151499577784238059763311e-04
            8.72773090460162492831841518636793e-01 3.02625797417711110692817300105162e-02 8.86625283151499577784238059763311e-04
            4.28753753691012928772607892824453e-01 1.46376163589685189675426713051820e-02 1.05970476604280001119151588540035e-03
            4.28753753691012928772607892824453e-01 5.56608629950018585219595479429699e-01 1.05970476604280001119151588540035e-03
            1.46376163589685189675426713051820e-02 4.28753753691012928772607892824453e-01 1.05970476604280001119151588540035e-03
            1.46376163589685189675426713051820e-02 5.56608629950018585219595479429699e-01 1.05970476604280001119151588540035e-03
            5.56608629950018585219595479429699e-01 4.28753753691012928772607892824453e-01 1.05970476604280001119151588540035e-03
            5.56608629950018585219595479429699e-01 1.46376163589685189675426713051820e-02 1.05970476604280001119151588540035e-03
            1.36853445326083039690345799499482e-01 1.36052509293985018912209739028185e-02 6.91132913699826203698939330877238e-04
            1.36853445326083039690345799499482e-01 8.49541303744518527807372265669983e-01 6.91132913699826203698939330877238e-04
            1.36052509293985018912209739028185e-02 1.36853445326083039690345799499482e-01 6.91132913699826203698939330877238e-04
            1.36052509293985018912209739028185e-02 8.49541303744518527807372265669983e-01 6.91132913699826203698939330877238e-04
            8.49541303744518527807372265669983e-01 1.36853445326083039690345799499482e-01 6.91132913699826203698939330877238e-04
            8.49541303744518527807372265669983e-01 1.36052509293985018912209739028185e-02 6.91132913699826203698939330877238e-04
            3.63561909099252844601579681693693e-01 2.47604062219227219476813672827120e-03 4.14389801137990615789719051065276e-04
            3.63561909099252844601579681693693e-01 6.33962050278554900550886941346107e-01 4.14389801137990615789719051065276e-04
            2.47604062219227219476813672827120e-03 3.63561909099252844601579681693693e-01 4.14389801137990615789719051065276e-04
            2.47604062219227219476813672827120e-03 6.33962050278554900550886941346107e-01 4.14389801137990615789719051065276e-04
            6.33962050278554900550886941346107e-01 3.63561909099252844601579681693693e-01 4.14389801137990615789719051065276e-04
            6.33962050278554900550886941346107e-01 2.47604062219227219476813672827120e-03 4.14389801137990615789719051065276e-04
            3.36143572327619710726764878927497e-01 1.42925973377929421692655864717381e-01 2.88046498518415659134683792785836e-03
            3.36143572327619710726764878927497e-01 5.20930454294450839825003640726209e-01 2.88046498518415659134683792785836e-03
            1.42925973377929421692655864717381e-01 3.36143572327619710726764878927497e-01 2.88046498518415659134683792785836e-03
            1.42925973377929421692655864717381e-01 5.20930454294450839825003640726209e-01 2.88046498518415659134683792785836e-03
            5.20930454294450839825003640726209e-01 3.36143572327619710726764878927497e-01 2.88046498518415659134683792785836e-03
            5.20930454294450839825003640726209e-01 1.42925973377929421692655864717381e-01 2.88046498518415659134683792785836e-03
            9.40988534499003353994694975881430e-02 2.42077774579726101231158530424636e-03 2.43320326789467253044210726109498e-04
            9.40988534499003353994694975881430e-02 9.03480368804302380603132860414917e-01 2.43320326789467253044210726109498e-04
            2.42077774579726101231158530424636e-03 9.40988534499003353994694975881430e-02 2.43320326789467253044210726109498e-04
            2.42077774579726101231158530424636e-03 9.03480368804302380603132860414917e-01 2.43320326789467253044210726109498e-04
            9.03480368804302380603132860414917e-01 9.40988534499003353994694975881430e-02 2.43320326789467253044210726109498e-04
            9.03480368804302380603132860414917e-01 2.42077774579726101231158530424636e-03 2.43320326789467253044210726109498e-04
            2.42710586669900529876997552491957e-01 1.95150339989513393890163683863648e-01 3.03058530224832849836458237291481e-03
            2.42710586669900529876997552491957e-01 5.62139073340586103988414379273308e-01 3.03058530224832849836458237291481e-03
            1.95150339989513393890163683863648e-01 2.42710586669900529876997552491957e-01 3.03058530224832849836458237291481e-03
            1.95150339989513393890163683863648e-01 5.62139073340586103988414379273308e-01 3.03058530224832849836458237291481e-03
            5.62139073340586103988414379273308e-01 2.42710586669900529876997552491957e-01 3.03058530224832849836458237291481e-03
            5.62139073340586103988414379273308e-01 1.95150339989513393890163683863648e-01 3.03058530224832849836458237291481e-03
            4.25238442831057605886257988458965e-01 6.40551018216570233754225682787364e-02 1.97958017650065988893293145167718e-03
            4.25238442831057605886257988458965e-01 5.10706455347285315227168212004472e-01 1.97958017650065988893293145167718e-03
            6.40551018216570233754225682787364e-02 4.25238442831057605886257988458965e-01 1.97958017650065988893293145167718e-03
            6.40551018216570233754225682787364e-02 5.10706455347285315227168212004472e-01 1.97958017650065988893293145167718e-03
            5.10706455347285315227168212004472e-01 4.25238442831057605886257988458965e-01 1.97958017650065988893293145167718e-03
            5.10706455347285315227168212004472e-01 6.40551018216570233754225682787364e-02 1.97958017650065988893293145167718e-03
            3.64278126974187321795994876083569e-01 1.31876112324832255040529815914851e-02 9.75765280742351098601516934394340e-04
            3.64278126974187321795994876083569e-01 6.22534261793329490863868613814702e-01 9.75765280742351098601516934394340e-04
            1.31876112324832255040529815914851e-02 3.64278126974187321795994876083569e-01 9.75765280742351098601516934394340e-04
            1.31876112324832255040529815914851e-02 6.22534261793329490863868613814702e-01 9.75765280742351098601516934394340e-04
            6.22534261793329490863868613814702e-01 3.64278126974187321795994876083569e-01 9.75765280742351098601516934394340e-04
            6.22534261793329490863868613814702e-01 1.31876112324832255040529815914851e-02 9.75765280742351098601516934394340e-04
            1.77663726489636492056334304834309e-01 6.18730821070126515981435488811258e-02 1.58793200978925058870772080155120e-03
            1.77663726489636492056334304834309e-01 7.60463191403350835528840434562881e-01 1.58793200978925058870772080155120e-03
            6.18730821070126515981435488811258e-02 1.77663726489636492056334304834309e-01 1.58793200978925058870772080155120e-03
            6.18730821070126515981435488811258e-02 7.60463191403350835528840434562881e-01 1.58793200978925058870772080155120e-03
            7.60463191403350835528840434562881e-01 1.77663726489636492056334304834309e-01 1.58793200978925058870772080155120e-03
            7.60463191403350835528840434562881e-01 6.18730821070126515981435488811258e-02 1.58793200978925058870772080155120e-03
            5.88456479492050557600357763021748e-02 2.61247234276882700845789919696927e-03 2.08884489201240227621708678285017e-04
            5.88456479492050557600357763021748e-02 9.38541879708026050010971630399581e-01 2.08884489201240227621708678285017e-04
            2.61247234276882700845789919696927e-03 5.88456479492050557600357763021748e-02 2.08884489201240227621708678285017e-04
            2.61247234276882700845789919696927e-03 9.38541879708026050010971630399581e-01 2.08884489201240227621708678285017e-04
            9.38541879708026050010971630399581e-01 5.88456479492050557600357763021748e-02 2.08884489201240227621708678285017e-04
            9.38541879708026050010971630399581e-01 2.61247234276882700845789919696927e-03 2.08884489201240227621708678285017e-04
            2.50099601909829272194940585904988e-01 3.41904035600949715578700249807298e-02 1.40189925650187536577961910921886e-03
            2.50099601909829272194940585904988e-01 7.15709994530075777063871100835968e-01 1.40189925650187536577961910921886e-03
            3.41904035600949715578700249807298e-02 2.50099601909829272194940585904988e-01 1.40189925650187536577961910921886e-03
            3.41904035600949715578700249807298e-02 7.15709994530075777063871100835968e-01 1.40189925650187536577961910921886e-03
            7.15709994530075777063871100835968e-01 2.50099601909829272194940585904988e-01 1.40189925650187536577961910921886e-03
            7.15709994530075777063871100835968e-01 3.41904035600949715578700249807298e-02 1.40189925650187536577961910921886e-03
            3.19725865365346195723361688578734e-02 2.65083133054744551210246683581317e-03 1.53716150062060003802719143628508e-04
            3.19725865365346195723361688578734e-02 9.65376582132917993028797809529351e-01 1.53716150062060003802719143628508e-04
            2.65083133054744551210246683581317e-03 3.19725865365346195723361688578734e-02 1.53716150062060003802719143628508e-04
            2.65083133054744551210246683581317e-03 9.65376582132917993028797809529351e-01 1.53716150062060003802719143628508e-04
            9.65376582132917993028797809529351e-01 3.19725865365346195723361688578734e-02 1.53716150062060003802719143628508e-04
            9.65376582132917993028797809529351e-01 2.65083133054744551210246683581317e-03 1.53716150062060003802719143628508e-04
            2.41723961357679040240498125058366e-01 2.59156908972399966165078843971514e-03 3.79023937257839244312912496681633e-04
            2.41723961357679040240498125058366e-01 7.55684469552596937980126767797628e-01 3.79023937257839244312912496681633e-04
            2.59156908972399966165078843971514e-03 2.41723961357679040240498125058366e-01 3.79023937257839244312912496681633e-04
            2.59156908972399966165078843971514e-03 7.55684469552596937980126767797628e-01 3.79023937257839244312912496681633e-04
            7.55684469552596937980126767797628e-01 2.41723961357679040240498125058366e-01 3.79023937257839244312912496681633e-04
            7.55684469552596937980126767797628e-01 2.59156908972399966165078843971514e-03 3.79023937257839244312912496681633e-04
            1.84404707439201814844764726331050e-01 1.42733464909807362258620244688245e-02 8.14280093595709324703901810238449e-04
            1.84404707439201814844764726331050e-01 8.01321946069817436786308917362476e-01 8.14280093595709324703901810238449e-04
            1.42733464909807362258620244688245e-02 1.84404707439201814844764726331050e-01 8.14280093595709324703901810238449e-04
            1.42733464909807362258620244688245e-02 8.01321946069817436786308917362476e-01 8.14280093595709324703901810238449e-04
            8.01321946069817436786308917362476e-01 1.84404707439201814844764726331050e-01 8.14280093595709324703901810238449e-04
            8.01321946069817436786308917362476e-01 1.42733464909807362258620244688245e-02 8.14280093595709324703901810238449e-04
            6.12563890152719517367962964726757e-02 1.38564076854345488015729515041130e-02 5.00790757565847816090709443415108e-04
            6.12563890152719517367962964726757e-02 9.24887203299293525482482891675318e-01 5.00790757565847816090709443415108e-04
            1.38564076854345488015729515041130e-02 6.12563890152719517367962964726757e-02 5.00790757565847816090709443415108e-04
            1.38564076854345488015729515041130e-02 9.24887203299293525482482891675318e-01 5.00790757565847816090709443415108e-04
            9.24887203299293525482482891675318e-01 6.12563890152719517367962964726757e-02 5.00790757565847816090709443415108e-04
            9.24887203299293525482482891675318e-01 1.38564076854345488015729515041130e-02 5.00790757565847816090709443415108e-04
            1.32571416922718544884052249699380e-02 2.49382626599786065904607035292884e-03 9.48517546105485443753979613035199e-05
            1.32571416922718544884052249699380e-02 9.84249032041730309572358237346634e-01 9.48517546105485443753979613035199e-05
            2.49382626599786065904607035292884e-03 1.32571416922718544884052249699380e-02 9.48517546105485443753979613035199e-05
            2.49382626599786065904607035292884e-03 9.84249032041730309572358237346634e-01 9.48517546105485443753979613035199e-05
            9.84249032041730309572358237346634e-01 1.32571416922718544884052249699380e-02 9.48517546105485443753979613035199e-05
            9.84249032041730309572358237346634e-01 2.49382626599786065904607035292884e-03 9.48517546105485443753979613035199e-05
            1.37973140081455225569584399636369e-01 1.00440931058371943351836819147138e-01 1.96129685941321589906505273859239e-03
            1.37973140081455225569584399636369e-01 7.61585928860172844956366589030949e-01 1.96129685941321589906505273859239e-03
            1.00440931058371943351836819147138e-01 1.37973140081455225569584399636369e-01 1.96129685941321589906505273859239e-03
            1.00440931058371943351836819147138e-01 7.61585928860172844956366589030949e-01 1.96129685941321589906505273859239e-03
            7.61585928860172844956366589030949e-01 1.37973140081455225569584399636369e-01 1.96129685941321589906505273859239e-03
            7.61585928860172844956366589030949e-01 1.00440931058371943351836819147138e-01 1.96129685941321589906505273859239e-03
            3.32056799273112990267264876820263e-02 1.36974423527236825043340573415662e-02 3.67632224535556766543270557079381e-04
            3.32056799273112990267264876820263e-02 9.53096877719965096531495873932727e-01 3.67632224535556766543270557079381e-04
            1.36974423527236825043340573415662e-02 3.32056799273112990267264876820263e-02 3.67632224535556766543270557079381e-04
            1.36974423527236825043340573415662e-02 9.53096877719965096531495873932727e-01 3.67632224535556766543270557079381e-04
            9.53096877719965096531495873932727e-01 3.32056799273112990267264876820263e-02 3.67632224535556766543270557079381e-04
            9.53096877719965096531495873932727e-01 1.36974423527236825043340573415662e-02 3.67632224535556766543270557079381e-04
            1.94466196781106465119037807198765e-01 3.41431958895440298973333881349390e-02 1.31387504913329416014744488450106e-03
            1.94466196781106465119037807198765e-01 7.71390607329349498044734900759067e-01 1.31387504913329416014744488450106e-03
            3.41431958895440298973333881349390e-02 1.94466196781106465119037807198765e-01 1.31387504913329416014744488450106e-03
            3.41431958895440298973333881349390e-02 7.71390607329349498044734900759067e-01 1.31387504913329416014744488450106e-03
            7.71390607329349498044734900759067e-01 1.94466196781106465119037807198765e-01 1.31387504913329416014744488450106e-03
            7.71390607329349498044734900759067e-01 3.41431958895440298973333881349390e-02 1.31387504913329416014744488450106e-03
            2.39482635085204537128689139535709e-01 1.38046252946412654344365833480879e-02 9.20816959513956543974255719575694e-04
            2.39482635085204537128689139535709e-01 7.46712739620154275499430696072523e-01 9.20816959513956543974255719575694e-04
            1.38046252946412654344365833480879e-02 2.39482635085204537128689139535709e-01 9.20816959513956543974255719575694e-04
            1.38046252946412654344365833480879e-02 7.46712739620154275499430696072523e-01 9.20816959513956543974255719575694e-04
            7.46712739620154275499430696072523e-01 2.39482635085204537128689139535709e-01 9.20816959513956543974255719575694e-04
            7.46712739620154275499430696072523e-01 1.38046252946412654344365833480879e-02 9.20816959513956543974255719575694e-04
            3.64554227776340777911201485039783e-01 1.86497214299555097438698680889502e-01 2.96251638633688099688789918673137e-03
            3.64554227776340777911201485039783e-01 4.48948557924104152405675449699629e-01 2.96251638633688099688789918673137e-03
            1.86497214299555097438698680889502e-01 3.64554227776340777911201485039783e-01 2.96251638633688099688789918673137e-03
            1.86497214299555097438698680889502e-01 4.48948557924104152405675449699629e-01 2.96251638633688099688789918673137e-03
            4.48948557924104152405675449699629e-01 3.64554227776340777911201485039783e-01 2.96251638633688099688789918673137e-03
            4.48948557924104152405675449699629e-01 1.86497214299555097438698680889502e-01 2.96251638633688099688789918673137e-03
            1.37356402366199809028657341514190e-01 2.54175936468376819851733294797214e-03 3.06169420407145201519505395282295e-04
            1.37356402366199809028657341514190e-01 8.60101838269116347746034989540931e-01 3.06169420407145201519505395282295e-04
            2.54175936468376819851733294797214e-03 1.37356402366199809028657341514190e-01 3.06169420407145201519505395282295e-04
            2.54175936468376819851733294797214e-03 8.60101838269116347746034989540931e-01 3.06169420407145201519505395282295e-04
            8.60101838269116347746034989540931e-01 1.37356402366199809028657341514190e-01 3.06169420407145201519505395282295e-04
            8.60101838269116347746034989540931e-01 2.54175936468376819851733294797214e-03 3.06169420407145201519505395282295e-04
            2.32232037318890388588599194008566e-01 6.30349337046809815587877778852999e-02 1.91248172423942944253560938250303e-03
            2.32232037318890388588599194008566e-01 7.04733028976428643730400835920591e-01 1.91248172423942944253560938250303e-03
            6.30349337046809815587877778852999e-02 2.32232037318890388588599194008566e-01 1.91248172423942944253560938250303e-03
            6.30349337046809815587877778852999e-02 7.04733028976428643730400835920591e-01 1.91248172423942944253560938250303e-03
            7.04733028976428643730400835920591e-01 2.32232037318890388588599194008566e-01 1.91248172423942944253560938250303e-03
            7.04733028976428643730400835920591e-01 6.30349337046809815587877778852999e-02 1.91248172423942944253560938250303e-03
            2.92661395072246866000398313190090e-01 6.38769125793660075274260634614620e-02 2.06415161110312878403605374444396e-03
            2.92661395072246866000398313190090e-01 6.43461692348387126472175623348448e-01 2.06415161110312878403605374444396e-03
            6.38769125793660075274260634614620e-02 2.92661395072246866000398313190090e-01 2.06415161110312878403605374444396e-03
            6.38769125793660075274260634614620e-02 6.43461692348387126472175623348448e-01 2.06415161110312878403605374444396e-03
            6.43461692348387126472175623348448e-01 2.92661395072246866000398313190090e-01 2.06415161110312878403605374444396e-03
            6.43461692348387126472175623348448e-01 6.38769125793660075274260634614620e-02 2.06415161110312878403605374444396e-03
            3.17648116398156710182831830024952e-01 2.41952845175475161898503984048148e-01 3.34601602087061583343241544241664e-03
            3.17648116398156710182831830024952e-01 4.40399038426368127918664185926900e-01 3.34601602087061583343241544241664e-03
            2.41952845175475161898503984048148e-01 3.17648116398156710182831830024952e-01 3.34601602087061583343241544241664e-03
            2.41952845175475161898503984048148e-01 4.40399038426368127918664185926900e-01 3.34601602087061583343241544241664e-03
            4.40399038426368127918664185926900e-01 3.17648116398156710182831830024952e-01 3.34601602087061583343241544241664e-03
            4.40399038426368127918664185926900e-01 2.41952845175475161898503984048148e-01 3.34601602087061583343241544241664e-03
            3.00301596615778321197609557202668e-01 1.45992712589763495162342721300774e-02 1.02364147369649050034468018566258e-03
            3.00301596615778321197609557202668e-01 6.85099132125245424695947349391645e-01 1.02364147369649050034468018566258e-03
            1.45992712589763495162342721300774e-02 3.00301596615778321197609557202668e-01 1.02364147369649050034468018566258e-03
            1.45992712589763495162342721300774e-02 6.85099132125245424695947349391645e-01 1.02364147369649050034468018566258e-03
            6.85099132125245424695947349391645e-01 3.00301596615778321197609557202668e-01 1.02364147369649050034468018566258e-03
            6.85099132125245424695947349391645e-01 1.45992712589763495162342721300774e-02 1.02364147369649050034468018566258e-03
            3.00561353716810975011952677959925e-01 2.81811459692608715435491184564398e-03 4.39007839582545515266093216766308e-04
            3.00561353716810975011952677959925e-01 6.96620531686262922654861995397368e-01 4.39007839582545515266093216766308e-04
            2.81811459692608715435491184564398e-03 3.00561353716810975011952677959925e-01 4.39007839582545515266093216766308e-04
            2.81811459692608715435491184564398e-03 6.96620531686262922654861995397368e-01 4.39007839582545515266093216766308e-04
            6.96620531686262922654861995397368e-01 3.00561353716810975011952677959925e-01 4.39007839582545515266093216766308e-04
            6.96620531686262922654861995397368e-01 2.81811459692608715435491184564398e-03 4.39007839582545515266093216766308e-04
            6.05239793987312202361295021546539e-02 3.41344995593797426169935249617993e-02 7.88892559272939180095574052131724e-04
            6.05239793987312202361295021546539e-02 9.05341521041889030207983068976318e-01 7.88892559272939180095574052131724e-04
            3.41344995593797426169935249617993e-02 6.05239793987312202361295021546539e-02 7.88892559272939180095574052131724e-04
            3.41344995593797426169935249617993e-02 9.05341521041889030207983068976318e-01 7.88892559272939180095574052131724e-04
            9.05341521041889030207983068976318e-01 6.05239793987312202361295021546539e-02 7.88892559272939180095574052131724e-04
            9.05341521041889030207983068976318e-01 3.41344995593797426169935249617993e-02 7.88892559272939180095574052131724e-04
            3.00785919233889365020928607918904e-01 1.93355477276418330490059815929271e-01 3.15371795549420404908280701761214e-03
            3.00785919233889365020928607918904e-01 5.05858603489692360000162807409652e-01 3.15371795549420404908280701761214e-03
            1.93355477276418330490059815929271e-01 3.00785919233889365020928607918904e-01 3.15371795549420404908280701761214e-03
            1.93355477276418330490059815929271e-01 5.05858603489692360000162807409652e-01 3.15371795549420404908280701761214e-03
            5.05858603489692360000162807409652e-01 3.00785919233889365020928607918904e-01 3.15371795549420404908280701761214e-03
            5.05858603489692360000162807409652e-01 1.93355477276418330490059815929271e-01 3.15371795549420404908280701761214e-03
            1.89550703977685197632041536053293e-01 9.73629697501508795109970151315792e-02 2.13631991052658405660924501034970e-03
            1.89550703977685197632041536053293e-01 7.13086326272163950612537064444041e-01 2.13631991052658405660924501034970e-03
            9.73629697501508795109970151315792e-02 1.89550703977685197632041536053293e-01 2.13631991052658405660924501034970e-03
            9.73629697501508795109970151315792e-02 7.13086326272163950612537064444041e-01 2.13631991052658405660924501034970e-03
            7.13086326272163950612537064444041e-01 1.89550703977685197632041536053293e-01 2.13631991052658405660924501034970e-03
            7.13086326272163950612537064444041e-01 9.73629697501508795109970151315792e-02 2.13631991052658405660924501034970e-03
            4.30139903362148912080442642036360e-01 2.78830240511142692441670654091013e-03 4.84918673119630056888434976869462e-04
            4.30139903362148912080442642036360e-01 5.67071794232739678776056280185003e-01 4.84918673119630056888434976869462e-04
            2.78830240511142692441670654091013e-03 4.30139903362148912080442642036360e-01 4.84918673119630056888434976869462e-04
            2.78830240511142692441670654091013e-03 5.67071794232739678776056280185003e-01 4.84918673119630056888434976869462e-04
            5.67071794232739678776056280185003e-01 4.30139903362148912080442642036360e-01 4.84918673119630056888434976869462e-04
            5.67071794232739678776056280185003e-01 2.78830240511142692441670654091013e-03 4.84918673119630056888434976869462e-04
            1.87018572217664685863880436045292e-01 2.75596244648371960922483836498031e-03 3.69009135228931571884314344345057e-04
            1.87018572217664685863880436045292e-01 8.10225465335851602333150367485359e-01 3.69009135228931571884314344345057e-04
            2.75596244648371960922483836498031e-03 1.87018572217664685863880436045292e-01 3.69009135228931571884314344345057e-04
            2.75596244648371960922483836498031e-03 8.10225465335851602333150367485359e-01 3.69009135228931571884314344345057e-04
            8.10225465335851602333150367485359e-01 1.87018572217664685863880436045292e-01 3.69009135228931571884314344345057e-04
            8.10225465335851602333150367485359e-01 2.75596244648371960922483836498031e-03 3.69009135228931571884314344345057e-04
            3.79189522509007448380913274377235e-01 9.78271071588677626484908955717401e-02 2.63094068006375966101151320231111e-03
            3.79189522509007448380913274377235e-01 5.22983370332124830603959253494395e-01 2.63094068006375966101151320231111e-03
            9.78271071588677626484908955717401e-02 3.79189522509007448380913274377235e-01 2.63094068006375966101151320231111e-03
            9.78271071588677626484908955717401e-02 5.22983370332124830603959253494395e-01 2.63094068006375966101151320231111e-03
            5.22983370332124830603959253494395e-01 3.79189522509007448380913274377235e-01 2.63094068006375966101151320231111e-03
            5.22983370332124830603959253494395e-01 9.78271071588677626484908955717401e-02 2.63094068006375966101151320231111e-03
            3.12966976982159250653126036922913e-01 9.95736856368709677411743541597389e-02 2.65013598749912638458114955142264e-03
            3.12966976982159250653126036922913e-01 5.87459337380969781605699608917348e-01 2.65013598749912638458114955142264e-03
            9.95736856368709677411743541597389e-02 3.12966976982159250653126036922913e-01 2.65013598749912638458114955142264e-03
            9.95736856368709677411743541597389e-02 5.87459337380969781605699608917348e-01 2.65013598749912638458114955142264e-03
            5.87459337380969781605699608917348e-01 3.12966976982159250653126036922913e-01 2.65013598749912638458114955142264e-03
            5.87459337380969781605699608917348e-01 9.95736856368709677411743541597389e-02 2.65013598749912638458114955142264e-03
            1.41804487017288982642071459849831e-01 3.41143516662648707549188031862286e-02 1.22790050513958497700739425084748e-03
            1.41804487017288982642071459849831e-01 8.24081161316446153541903640871169e-01 1.22790050513958497700739425084748e-03
            3.41143516662648707549188031862286e-02 1.41804487017288982642071459849831e-01 1.22790050513958497700739425084748e-03
            3.41143516662648707549188031862286e-02 8.24081161316446153541903640871169e-01 1.22790050513958497700739425084748e-03
            8.24081161316446153541903640871169e-01 1.41804487017288982642071459849831e-01 1.22790050513958497700739425084748e-03
            8.24081161316446153541903640871169e-01 3.41143516662648707549188031862286e-02 1.22790050513958497700739425084748e-03
            2.48181460015588495338079155771993e-01 1.00497443272928421720280312001705e-01 2.52780519757884276563220105060736e-03
            2.48181460015588495338079155771993e-01 6.51321096711483082941640532226302e-01 2.52780519757884276563220105060736e-03
            1.00497443272928421720280312001705e-01 2.48181460015588495338079155771993e-01 2.52780519757884276563220105060736e-03
            1.00497443272928421720280312001705e-01 6.51321096711483082941640532226302e-01 2.52780519757884276563220105060736e-03
            6.51321096711483082941640532226302e-01 2.48181460015588495338079155771993e-01 2.52780519757884276563220105060736e-03
            6.51321096711483082941640532226302e-01 1.00497443272928421720280312001705e-01 2.52780519757884276563220105060736e-03
            3.58797835407998699785991902899696e-01 6.16738036965516905896222965566267e-02 2.21709548456510601235902591099602e-03
            3.58797835407998699785991902899696e-01 5.79528360895449567991022377100307e-01 2.21709548456510601235902591099602e-03
            6.16738036965516905896222965566267e-02 3.58797835407998699785991902899696e-01 2.21709548456510601235902591099602e-03
            6.16738036965516905896222965566267e-02 5.79528360895449567991022377100307e-01 2.21709548456510601235902591099602e-03
            5.79528360895449567991022377100307e-01 3.58797835407998699785991902899696e-01 2.21709548456510601235902591099602e-03
            5.79528360895449567991022377100307e-01 6.16738036965516905896222965566267e-02 2.21709548456510601235902591099602e-03
            ];
        
    case  44
        
        % ALG. DEG.:   44
        % PTS CARD.:  354
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.4e-15
        
        xyw=[
            1.71230757996369982974760315386220e-03 1.71230757996369982974760315386220e-03 2.21520454570737307072644112215798e-05
            1.71230757996369982974760315386220e-03 9.96575384840072575620695261022775e-01 2.21520454570737307072644112215798e-05
            9.96575384840072575620695261022775e-01 1.71230757996369982974760315386220e-03 2.21520454570737307072644112215798e-05
            2.91673514653004867236418817810772e-02 2.91673514653004867236418817810772e-02 5.61488691411857524481143766337254e-04
            2.91673514653004867236418817810772e-02 9.41665297069399054308291852066759e-01 5.61488691411857524481143766337254e-04
            9.41665297069399054308291852066759e-01 2.91673514653004867236418817810772e-02 5.61488691411857524481143766337254e-04
            4.95499879123024178362300062872237e-02 4.95499879123024178362300062872237e-02 8.72575782766472408426283724480754e-04
            4.95499879123024178362300062872237e-02 9.00900024175395164327539987425553e-01 8.72575782766472408426283724480754e-04
            9.00900024175395164327539987425553e-01 4.95499879123024178362300062872237e-02 8.72575782766472408426283724480754e-04
            4.87723980583153360335302295425208e-01 4.87723980583153360335302295425208e-01 1.36482635824496534998617747191929e-03
            4.87723980583153360335302295425208e-01 2.45520388336932793293954091495834e-02 1.36482635824496534998617747191929e-03
            2.45520388336932793293954091495834e-02 4.87723980583153360335302295425208e-01 1.36482635824496534998617747191929e-03
            4.96061804465474931458146556906286e-01 4.96061804465474931458146556906286e-01 6.02253006705961600394938493252539e-04
            4.96061804465474931458146556906286e-01 7.87639106905013708370688618742861e-03 6.02253006705961600394938493252539e-04
            7.87639106905013708370688618742861e-03 4.96061804465474931458146556906286e-01 6.02253006705961600394938493252539e-04
            7.65961156051737368510856640568818e-02 7.65961156051737368510856640568818e-02 1.18289665538962355216534572122100e-03
            7.65961156051737368510856640568818e-02 8.46807768789652470786677440628409e-01 1.18289665538962355216534572122100e-03
            8.46807768789652470786677440628409e-01 7.65961156051737368510856640568818e-02 1.18289665538962355216534572122100e-03
            4.75951846052771243567036663080216e-01 4.75951846052771243567036663080216e-01 1.75559713417060541534864270829530e-03
            4.75951846052771243567036663080216e-01 4.80963078944575128659266738395672e-02 1.75559713417060541534864270829530e-03
            4.80963078944575128659266738395672e-02 4.75951846052771243567036663080216e-01 1.75559713417060541534864270829530e-03
            1.09095541957237654462353759754478e-01 1.09095541957237654462353759754478e-01 1.47173402359788551424513158849550e-03
            1.09095541957237654462353759754478e-01 7.81808916085524718830868096119957e-01 1.47173402359788551424513158849550e-03
            7.81808916085524718830868096119957e-01 1.09095541957237654462353759754478e-01 1.47173402359788551424513158849550e-03
            2.31247003693840935678593950797222e-01 2.31247003693840935678593950797222e-01 3.36162144923654376479915661946052e-03
            2.31247003693840935678593950797222e-01 5.37505992612318128642812098405557e-01 3.36162144923654376479915661946052e-03
            5.37505992612318128642812098405557e-01 2.31247003693840935678593950797222e-01 3.36162144923654376479915661946052e-03
            1.95908198223650492630909525360039e-01 1.95908198223650492630909525360039e-01 2.48992940625945846644384573664865e-03
            1.95908198223650492630909525360039e-01 6.08183603552699070249332180537749e-01 2.48992940625945846644384573664865e-03
            6.08183603552699070249332180537749e-01 1.95908198223650492630909525360039e-01 2.48992940625945846644384573664865e-03
            4.22321808207151372371868092159275e-01 4.22321808207151372371868092159275e-01 2.46041862262338191166821665945008e-03
            4.22321808207151372371868092159275e-01 1.55356383585697255256263815681450e-01 2.46041862262338191166821665945008e-03
            1.55356383585697255256263815681450e-01 4.22321808207151372371868092159275e-01 2.46041862262338191166821665945008e-03
            4.54881749959505921054869759245776e-01 4.54881749959505921054869759245776e-01 2.51073751429879829175084715586763e-03
            4.54881749959505921054869759245776e-01 9.02365000809881578902604815084487e-02 2.51073751429879829175084715586763e-03
            9.02365000809881578902604815084487e-02 4.54881749959505921054869759245776e-01 2.51073751429879829175084715586763e-03
            1.43900119540420812214165380282793e-01 1.43900119540420812214165380282793e-01 2.22329662038065400606057231414070e-03
            1.43900119540420812214165380282793e-01 7.12199760919158375571669239434414e-01 2.22329662038065400606057231414070e-03
            7.12199760919158375571669239434414e-01 1.43900119540420812214165380282793e-01 2.22329662038065400606057231414070e-03
            4.02340447854735383526048053681734e-01 4.02340447854735383526048053681734e-01 2.32330638229219987520024481852943e-03
            4.02340447854735383526048053681734e-01 1.95319104290529232947903892636532e-01 2.32330638229219987520024481852943e-03
            1.95319104290529232947903892636532e-01 4.02340447854735383526048053681734e-01 2.32330638229219987520024481852943e-03
            2.94295765690833932559655750083039e-01 2.94295765690833932559655750083039e-01 3.21273602508716097161944880156170e-03
            2.94295765690833932559655750083039e-01 4.11408468618332134880688499833923e-01 3.21273602508716097161944880156170e-03
            4.11408468618332134880688499833923e-01 2.94295765690833932559655750083039e-01 3.21273602508716097161944880156170e-03
            3.51615788795768935415253508836031e-01 3.51615788795768935415253508836031e-01 3.13279394181341245065319789375735e-03
            3.51615788795768935415253508836031e-01 2.96768422408462129169492982327938e-01 3.13279394181341245065319789375735e-03
            2.96768422408462129169492982327938e-01 3.51615788795768935415253508836031e-01 3.13279394181341245065319789375735e-03
            1.97551676157197361738404595143948e-02 1.07567195566842938037055432687339e-02 2.05987823218478822723948029604912e-04
            1.97551676157197361738404595143948e-02 9.69488112827595949205772285495186e-01 2.05987823218478822723948029604912e-04
            1.07567195566842938037055432687339e-02 1.97551676157197361738404595143948e-02 2.05987823218478822723948029604912e-04
            1.07567195566842938037055432687339e-02 9.69488112827595949205772285495186e-01 2.05987823218478822723948029604912e-04
            9.69488112827595949205772285495186e-01 1.97551676157197361738404595143948e-02 2.05987823218478822723948029604912e-04
            9.69488112827595949205772285495186e-01 1.07567195566842938037055432687339e-02 2.05987823218478822723948029604912e-04
            9.95165813818446182148491629959608e-02 7.80585899861933429737248513191616e-04 1.36620423001840941628834413812399e-04
            9.95165813818446182148491629959608e-02 8.99702832718293432634482087451033e-01 1.36620423001840941628834413812399e-04
            7.80585899861933429737248513191616e-04 9.95165813818446182148491629959608e-02 1.36620423001840941628834413812399e-04
            7.80585899861933429737248513191616e-04 8.99702832718293432634482087451033e-01 1.36620423001840941628834413812399e-04
            8.99702832718293432634482087451033e-01 9.95165813818446182148491629959608e-02 1.36620423001840941628834413812399e-04
            8.99702832718293432634482087451033e-01 7.80585899861933429737248513191616e-04 1.36620423001840941628834413812399e-04
            1.53494544755213874198673806859006e-01 1.93833291279059036736620580398949e-03 2.69664221964948346010881863321629e-04
            1.53494544755213874198673806859006e-01 8.44567122331995601136611639958574e-01 2.69664221964948346010881863321629e-04
            1.93833291279059036736620580398949e-03 1.53494544755213874198673806859006e-01 2.69664221964948346010881863321629e-04
            1.93833291279059036736620580398949e-03 8.44567122331995601136611639958574e-01 2.69664221964948346010881863321629e-04
            8.44567122331995601136611639958574e-01 1.53494544755213874198673806859006e-01 2.69664221964948346010881863321629e-04
            8.44567122331995601136611639958574e-01 1.93833291279059036736620580398949e-03 2.69664221964948346010881863321629e-04
            4.60607452003998885103897009685170e-01 1.78614248052541058943420981819372e-03 3.35489485291019079969687677333923e-04
            4.60607452003998885103897009685170e-01 5.37606405515475671563763171434402e-01 3.35489485291019079969687677333923e-04
            1.78614248052541058943420981819372e-03 4.60607452003998885103897009685170e-01 3.35489485291019079969687677333923e-04
            1.78614248052541058943420981819372e-03 5.37606405515475671563763171434402e-01 3.35489485291019079969687677333923e-04
            5.37606405515475671563763171434402e-01 4.60607452003998885103897009685170e-01 3.35489485291019079969687677333923e-04
            5.37606405515475671563763171434402e-01 1.78614248052541058943420981819372e-03 3.35489485291019079969687677333923e-04
            2.96420604264443915409898977486591e-02 1.75422517105275517501628002037251e-03 1.12965768056969912335948946768838e-04
            2.96420604264443915409898977486591e-02 9.68603714402502924407656337280059e-01 1.12965768056969912335948946768838e-04
            1.75422517105275517501628002037251e-03 2.96420604264443915409898977486591e-02 1.12965768056969912335948946768838e-04
            1.75422517105275517501628002037251e-03 9.68603714402502924407656337280059e-01 1.12965768056969912335948946768838e-04
            9.68603714402502924407656337280059e-01 2.96420604264443915409898977486591e-02 1.12965768056969912335948946768838e-04
            9.68603714402502924407656337280059e-01 1.75422517105275517501628002037251e-03 1.12965768056969912335948946768838e-04
            3.33840624824811804671753634465858e-01 5.17294259660069986710151379583067e-04 1.61033154709168695086635159086086e-04
            3.33840624824811804671753634465858e-01 6.65642080915528122631030782940798e-01 1.61033154709168695086635159086086e-04
            5.17294259660069986710151379583067e-04 3.33840624824811804671753634465858e-01 1.61033154709168695086635159086086e-04
            5.17294259660069986710151379583067e-04 6.65642080915528122631030782940798e-01 1.61033154709168695086635159086086e-04
            6.65642080915528122631030782940798e-01 3.33840624824811804671753634465858e-01 1.61033154709168695086635159086086e-04
            6.65642080915528122631030782940798e-01 5.17294259660069986710151379583067e-04 1.61033154709168695086635159086086e-04
            1.19426687227411967850976282079500e-01 7.54587092570611038533368031266946e-03 4.83883651397566259995397519588778e-04
            1.19426687227411967850976282079500e-01 8.73027441846881857578921426465968e-01 4.83883651397566259995397519588778e-04
            7.54587092570611038533368031266946e-03 1.19426687227411967850976282079500e-01 4.83883651397566259995397519588778e-04
            7.54587092570611038533368031266946e-03 8.73027441846881857578921426465968e-01 4.83883651397566259995397519588778e-04
            8.73027441846881857578921426465968e-01 1.19426687227411967850976282079500e-01 4.83883651397566259995397519588778e-04
            8.73027441846881857578921426465968e-01 7.54587092570611038533368031266946e-03 4.83883651397566259995397519588778e-04
            1.73311806699428916367367037310032e-01 1.14925464528052804352720173142188e-02 6.73332446090880174308090921897474e-04
            1.73311806699428916367367037310032e-01 8.15195646847765797993190517445328e-01 6.73332446090880174308090921897474e-04
            1.14925464528052804352720173142188e-02 1.73311806699428916367367037310032e-01 6.73332446090880174308090921897474e-04
            1.14925464528052804352720173142188e-02 8.15195646847765797993190517445328e-01 6.73332446090880174308090921897474e-04
            8.15195646847765797993190517445328e-01 1.73311806699428916367367037310032e-01 6.73332446090880174308090921897474e-04
            8.15195646847765797993190517445328e-01 1.14925464528052804352720173142188e-02 6.73332446090880174308090921897474e-04
            5.93978633857566162834729084352148e-02 2.42629677211045335916139720211504e-02 6.68754141990617208789193792739525e-04
            5.93978633857566162834729084352148e-02 9.16339168893138822369337503914721e-01 6.68754141990617208789193792739525e-04
            2.42629677211045335916139720211504e-02 5.93978633857566162834729084352148e-02 6.68754141990617208789193792739525e-04
            2.42629677211045335916139720211504e-02 9.16339168893138822369337503914721e-01 6.68754141990617208789193792739525e-04
            9.16339168893138822369337503914721e-01 5.93978633857566162834729084352148e-02 6.68754141990617208789193792739525e-04
            9.16339168893138822369337503914721e-01 2.42629677211045335916139720211504e-02 6.68754141990617208789193792739525e-04
            2.59446164627843145478180986174266e-01 9.79769330245964208125908356805667e-03 6.04714018344115408765127028800634e-04
            2.59446164627843145478180986174266e-01 7.30756142069697189889154742559185e-01 6.04714018344115408765127028800634e-04
            9.79769330245964208125908356805667e-03 2.59446164627843145478180986174266e-01 6.04714018344115408765127028800634e-04
            9.79769330245964208125908356805667e-03 7.30756142069697189889154742559185e-01 6.04714018344115408765127028800634e-04
            7.30756142069697189889154742559185e-01 2.59446164627843145478180986174266e-01 6.04714018344115408765127028800634e-04
            7.30756142069697189889154742559185e-01 9.79769330245964208125908356805667e-03 6.04714018344115408765127028800634e-04
            2.69664879531117784861038444432779e-01 1.89915082820833660409709509053755e-03 2.98688070880941223140397910995603e-04
            2.69664879531117784861038444432779e-01 7.28435969640673808278563683415996e-01 2.98688070880941223140397910995603e-04
            1.89915082820833660409709509053755e-03 2.69664879531117784861038444432779e-01 2.98688070880941223140397910995603e-04
            1.89915082820833660409709509053755e-03 7.28435969640673808278563683415996e-01 2.98688070880941223140397910995603e-04
            7.28435969640673808278563683415996e-01 2.69664879531117784861038444432779e-01 2.98688070880941223140397910995603e-04
            7.28435969640673808278563683415996e-01 1.89915082820833660409709509053755e-03 2.98688070880941223140397910995603e-04
            3.23428410410956435061535785280284e-01 7.67467985614434661245653046535153e-03 7.34100531866784780828105727579214e-04
            3.23428410410956435061535785280284e-01 6.68896909732899280776052819419419e-01 7.34100531866784780828105727579214e-04
            7.67467985614434661245653046535153e-03 3.23428410410956435061535785280284e-01 7.34100531866784780828105727579214e-04
            7.67467985614434661245653046535153e-03 6.68896909732899280776052819419419e-01 7.34100531866784780828105727579214e-04
            6.68896909732899280776052819419419e-01 3.23428410410956435061535785280284e-01 7.34100531866784780828105727579214e-04
            6.68896909732899280776052819419419e-01 7.67467985614434661245653046535153e-03 7.34100531866784780828105727579214e-04
            2.09397609435192477711495939729502e-01 2.70720430776859224270003423384878e-03 4.00527242600620333751876289696270e-04
            2.09397609435192477711495939729502e-01 7.87895186257038915300654480233788e-01 4.00527242600620333751876289696270e-04
            2.70720430776859224270003423384878e-03 2.09397609435192477711495939729502e-01 4.00527242600620333751876289696270e-04
            2.70720430776859224270003423384878e-03 7.87895186257038915300654480233788e-01 4.00527242600620333751876289696270e-04
            7.87895186257038915300654480233788e-01 2.09397609435192477711495939729502e-01 4.00527242600620333751876289696270e-04
            7.87895186257038915300654480233788e-01 2.70720430776859224270003423384878e-03 4.00527242600620333751876289696270e-04
            1.07937373867395408622016361732676e-02 2.70639655716989335765276791789802e-03 9.27835211688662648155806467187290e-05
            1.07937373867395408622016361732676e-02 9.86499866056090635169084634981118e-01 9.27835211688662648155806467187290e-05
            2.70639655716989335765276791789802e-03 1.07937373867395408622016361732676e-02 9.27835211688662648155806467187290e-05
            2.70639655716989335765276791789802e-03 9.86499866056090635169084634981118e-01 9.27835211688662648155806467187290e-05
            9.86499866056090635169084634981118e-01 1.07937373867395408622016361732676e-02 9.27835211688662648155806467187290e-05
            9.86499866056090635169084634981118e-01 2.70639655716989335765276791789802e-03 9.27835211688662648155806467187290e-05
            4.16577600266789249694454611017136e-02 1.04627340296394866214768626377918e-02 3.60711013213061229294026377445448e-04
            4.16577600266789249694454611017136e-02 9.47879505943681532897926445002668e-01 3.60711013213061229294026377445448e-04
            1.04627340296394866214768626377918e-02 4.16577600266789249694454611017136e-02 3.60711013213061229294026377445448e-04
            1.04627340296394866214768626377918e-02 9.47879505943681532897926445002668e-01 3.60711013213061229294026377445448e-04
            9.47879505943681532897926445002668e-01 4.16577600266789249694454611017136e-02 3.60711013213061229294026377445448e-04
            9.47879505943681532897926445002668e-01 1.04627340296394866214768626377918e-02 3.60711013213061229294026377445448e-04
            1.39300774418482237182459471114271e-01 2.15128602849198879987113741663052e-02 7.66083362023713934814417303442724e-04
            1.39300774418482237182459471114271e-01 8.39186365296597847063253539090510e-01 7.66083362023713934814417303442724e-04
            2.15128602849198879987113741663052e-02 1.39300774418482237182459471114271e-01 7.66083362023713934814417303442724e-04
            2.15128602849198879987113741663052e-02 8.39186365296597847063253539090510e-01 7.66083362023713934814417303442724e-04
            8.39186365296597847063253539090510e-01 1.39300774418482237182459471114271e-01 7.66083362023713934814417303442724e-04
            8.39186365296597847063253539090510e-01 2.15128602849198879987113741663052e-02 7.66083362023713934814417303442724e-04
            9.71407574555290531659323960411712e-02 2.28245356349519729066699369468552e-02 7.54513310214146746754404482260270e-04
            9.71407574555290531659323960411712e-02 8.80034706909518904538458627939690e-01 7.54513310214146746754404482260270e-04
            2.28245356349519729066699369468552e-02 9.71407574555290531659323960411712e-02 7.54513310214146746754404482260270e-04
            2.28245356349519729066699369468552e-02 8.80034706909518904538458627939690e-01 7.54513310214146746754404482260270e-04
            8.80034706909518904538458627939690e-01 9.71407574555290531659323960411712e-02 7.54513310214146746754404482260270e-04
            8.80034706909518904538458627939690e-01 2.28245356349519729066699369468552e-02 7.54513310214146746754404482260270e-04
            1.51124809738970578010253120737616e-01 1.00088530304790021552285850248154e-01 1.70086123647450698607996866229541e-03
            1.51124809738970578010253120737616e-01 7.48786659956239386559673221199773e-01 1.70086123647450698607996866229541e-03
            1.00088530304790021552285850248154e-01 1.51124809738970578010253120737616e-01 1.70086123647450698607996866229541e-03
            1.00088530304790021552285850248154e-01 7.48786659956239386559673221199773e-01 1.70086123647450698607996866229541e-03
            7.48786659956239386559673221199773e-01 1.51124809738970578010253120737616e-01 1.70086123647450698607996866229541e-03
            7.48786659956239386559673221199773e-01 1.00088530304790021552285850248154e-01 1.70086123647450698607996866229541e-03
            3.93932395535154467047789239586564e-01 3.15956953663550782485103241015167e-03 5.17994081903192481186604112508576e-04
            3.93932395535154467047789239586564e-01 6.02908034928209990432890208467143e-01 5.17994081903192481186604112508576e-04
            3.15956953663550782485103241015167e-03 3.93932395535154467047789239586564e-01 5.17994081903192481186604112508576e-04
            3.15956953663550782485103241015167e-03 6.02908034928209990432890208467143e-01 5.17994081903192481186604112508576e-04
            6.02908034928209990432890208467143e-01 3.93932395535154467047789239586564e-01 5.17994081903192481186604112508576e-04
            6.02908034928209990432890208467143e-01 3.15956953663550782485103241015167e-03 5.17994081903192481186604112508576e-04
            1.27637459417324766741685948545637e-01 4.35675752808351648970308644948091e-02 1.18874384638993006550744535587683e-03
            1.27637459417324766741685948545637e-01 8.28794965301840047544601475237869e-01 1.18874384638993006550744535587683e-03
            4.35675752808351648970308644948091e-02 1.27637459417324766741685948545637e-01 1.18874384638993006550744535587683e-03
            4.35675752808351648970308644948091e-02 8.28794965301840047544601475237869e-01 1.18874384638993006550744535587683e-03
            8.28794965301840047544601475237869e-01 1.27637459417324766741685948545637e-01 1.18874384638993006550744535587683e-03
            8.28794965301840047544601475237869e-01 4.35675752808351648970308644948091e-02 1.18874384638993006550744535587683e-03
            2.15342785917052303990004702427541e-01 6.28153762458395631407270798263198e-02 1.69769561709233744901947904537565e-03
            2.15342785917052303990004702427541e-01 7.21841837837108202258207256818423e-01 1.69769561709233744901947904537565e-03
            6.28153762458395631407270798263198e-02 2.15342785917052303990004702427541e-01 1.69769561709233744901947904537565e-03
            6.28153762458395631407270798263198e-02 7.21841837837108202258207256818423e-01 1.69769561709233744901947904537565e-03
            7.21841837837108202258207256818423e-01 2.15342785917052303990004702427541e-01 1.69769561709233744901947904537565e-03
            7.21841837837108202258207256818423e-01 6.28153762458395631407270798263198e-02 1.69769561709233744901947904537565e-03
            1.80523887256028525705175979965134e-01 3.46911866169311577934308843396138e-02 1.35204419124090890504119588655385e-03
            1.80523887256028525705175979965134e-01 7.84784926127040316501393135695253e-01 1.35204419124090890504119588655385e-03
            3.46911866169311577934308843396138e-02 1.80523887256028525705175979965134e-01 1.35204419124090890504119588655385e-03
            3.46911866169311577934308843396138e-02 7.84784926127040316501393135695253e-01 1.35204419124090890504119588655385e-03
            7.84784926127040316501393135695253e-01 1.80523887256028525705175979965134e-01 1.35204419124090890504119588655385e-03
            7.84784926127040316501393135695253e-01 3.46911866169311577934308843396138e-02 1.35204419124090890504119588655385e-03
            3.70697478749728304148192137290607e-01 1.59521357680311760829372502712431e-02 1.00850169514564371071874138152680e-03
            3.70697478749728304148192137290607e-01 6.13350385482240412216015101876110e-01 1.00850169514564371071874138152680e-03
            1.59521357680311760829372502712431e-02 3.70697478749728304148192137290607e-01 1.00850169514564371071874138152680e-03
            1.59521357680311760829372502712431e-02 6.13350385482240412216015101876110e-01 1.00850169514564371071874138152680e-03
            6.13350385482240412216015101876110e-01 3.70697478749728304148192137290607e-01 1.00850169514564371071874138152680e-03
            6.13350385482240412216015101876110e-01 1.59521357680311760829372502712431e-02 1.00850169514564371071874138152680e-03
            2.38904346634593867104712217042106e-01 3.65656909809639746322673659051361e-02 1.39594056621288411558501341147576e-03
            2.38904346634593867104712217042106e-01 7.24529962384442116629656993609387e-01 1.39594056621288411558501341147576e-03
            3.65656909809639746322673659051361e-02 2.38904346634593867104712217042106e-01 1.39594056621288411558501341147576e-03
            3.65656909809639746322673659051361e-02 7.24529962384442116629656993609387e-01 1.39594056621288411558501341147576e-03
            7.24529962384442116629656993609387e-01 2.38904346634593867104712217042106e-01 1.39594056621288411558501341147576e-03
            7.24529962384442116629656993609387e-01 3.65656909809639746322673659051361e-02 1.39594056621288411558501341147576e-03
            4.03544504764916933670093612818164e-01 1.22867248406971205088034082564263e-01 2.15022943286524628042633011659746e-03
            4.03544504764916933670093612818164e-01 4.73588246828111902875235728060943e-01 2.15022943286524628042633011659746e-03
            1.22867248406971205088034082564263e-01 4.03544504764916933670093612818164e-01 2.15022943286524628042633011659746e-03
            1.22867248406971205088034082564263e-01 4.73588246828111902875235728060943e-01 2.15022943286524628042633011659746e-03
            4.73588246828111902875235728060943e-01 4.03544504764916933670093612818164e-01 2.15022943286524628042633011659746e-03
            4.73588246828111902875235728060943e-01 1.22867248406971205088034082564263e-01 2.15022943286524628042633011659746e-03
            2.51088618401061669516138863400556e-01 1.22061192033510804155760354205995e-01 2.49192826459687828688505106811135e-03
            2.51088618401061669516138863400556e-01 6.26850189565427484694737358950078e-01 2.49192826459687828688505106811135e-03
            1.22061192033510804155760354205995e-01 2.51088618401061669516138863400556e-01 2.49192826459687828688505106811135e-03
            1.22061192033510804155760354205995e-01 6.26850189565427484694737358950078e-01 2.49192826459687828688505106811135e-03
            6.26850189565427484694737358950078e-01 2.51088618401061669516138863400556e-01 2.49192826459687828688505106811135e-03
            6.26850189565427484694737358950078e-01 1.22061192033510804155760354205995e-01 2.49192826459687828688505106811135e-03
            3.10092460353001919681759090963169e-01 8.78278870328067923223258617326792e-02 2.32664813783349305395531381179808e-03
            3.10092460353001919681759090963169e-01 6.02079652614191274118127239489695e-01 2.32664813783349305395531381179808e-03
            8.78278870328067923223258617326792e-02 3.10092460353001919681759090963169e-01 2.32664813783349305395531381179808e-03
            8.78278870328067923223258617326792e-02 6.02079652614191274118127239489695e-01 2.32664813783349305395531381179808e-03
            6.02079652614191274118127239489695e-01 3.10092460353001919681759090963169e-01 2.32664813783349305395531381179808e-03
            6.02079652614191274118127239489695e-01 8.78278870328067923223258617326792e-02 2.32664813783349305395531381179808e-03
            4.17188937068111109773838052205974e-01 6.56811030561964964702426073017705e-02 2.06393802112825059530409887997848e-03
            4.17188937068111109773838052205974e-01 5.17129959875692435389282763935626e-01 2.06393802112825059530409887997848e-03
            6.56811030561964964702426073017705e-02 4.17188937068111109773838052205974e-01 2.06393802112825059530409887997848e-03
            6.56811030561964964702426073017705e-02 5.17129959875692435389282763935626e-01 2.06393802112825059530409887997848e-03
            5.17129959875692435389282763935626e-01 4.17188937068111109773838052205974e-01 2.06393802112825059530409887997848e-03
            5.17129959875692435389282763935626e-01 6.56811030561964964702426073017705e-02 2.06393802112825059530409887997848e-03
            8.42771975158688041451782169133367e-02 4.57201471848305632250664132243401e-02 1.02891088117624531980687230259264e-03
            8.42771975158688041451782169133367e-02 8.70002655299300653446437081584008e-01 1.02891088117624531980687230259264e-03
            4.57201471848305632250664132243401e-02 8.42771975158688041451782169133367e-02 1.02891088117624531980687230259264e-03
            4.57201471848305632250664132243401e-02 8.70002655299300653446437081584008e-01 1.02891088117624531980687230259264e-03
            8.70002655299300653446437081584008e-01 8.42771975158688041451782169133367e-02 1.02891088117624531980687230259264e-03
            8.70002655299300653446437081584008e-01 4.57201471848305632250664132243401e-02 1.02891088117624531980687230259264e-03
            2.87517281890495191021273058140650e-01 5.28034108950670294513329849905858e-02 1.94691022921440444817808845812124e-03
            2.87517281890495191021273058140650e-01 6.59679307214437793405181764683221e-01 1.94691022921440444817808845812124e-03
            5.28034108950670294513329849905858e-02 2.87517281890495191021273058140650e-01 1.94691022921440444817808845812124e-03
            5.28034108950670294513329849905858e-02 6.59679307214437793405181764683221e-01 1.94691022921440444817808845812124e-03
            6.59679307214437793405181764683221e-01 2.87517281890495191021273058140650e-01 1.94691022921440444817808845812124e-03
            6.59679307214437793405181764683221e-01 5.28034108950670294513329849905858e-02 1.94691022921440444817808845812124e-03
            3.42142184906992441462136866903165e-01 3.32627075502785821137230470867507e-02 1.38538069827777053297235276829724e-03
            3.42142184906992441462136866903165e-01 6.24595107542728955607458374288399e-01 1.38538069827777053297235276829724e-03
            3.32627075502785821137230470867507e-02 3.42142184906992441462136866903165e-01 1.38538069827777053297235276829724e-03
            3.32627075502785821137230470867507e-02 6.24595107542728955607458374288399e-01 1.38538069827777053297235276829724e-03
            6.24595107542728955607458374288399e-01 3.42142184906992441462136866903165e-01 1.38538069827777053297235276829724e-03
            6.24595107542728955607458374288399e-01 3.32627075502785821137230470867507e-02 1.38538069827777053297235276829724e-03
            2.90781335627484260886177480642800e-01 2.30909511650298354679122780908074e-02 1.24007199925338091987814248540190e-03
            2.90781335627484260886177480642800e-01 6.86127713207486000790424895967590e-01 1.24007199925338091987814248540190e-03
            2.30909511650298354679122780908074e-02 2.90781335627484260886177480642800e-01 1.24007199925338091987814248540190e-03
            2.30909511650298354679122780908074e-02 6.86127713207486000790424895967590e-01 1.24007199925338091987814248540190e-03
            6.86127713207486000790424895967590e-01 2.90781335627484260886177480642800e-01 1.24007199925338091987814248540190e-03
            6.86127713207486000790424895967590e-01 2.30909511650298354679122780908074e-02 1.24007199925338091987814248540190e-03
            7.77385297161002769561477521165216e-02 8.41212288864582984071471827292044e-03 4.31451090416773528225108380595998e-04
            7.77385297161002769561477521165216e-02 9.13849347395253919223989669262664e-01 4.31451090416773528225108380595998e-04
            8.41212288864582984071471827292044e-03 7.77385297161002769561477521165216e-02 4.31451090416773528225108380595998e-04
            8.41212288864582984071471827292044e-03 9.13849347395253919223989669262664e-01 4.31451090416773528225108380595998e-04
            9.13849347395253919223989669262664e-01 7.77385297161002769561477521165216e-02 4.31451090416773528225108380595998e-04
            9.13849347395253919223989669262664e-01 8.41212288864582984071471827292044e-03 4.31451090416773528225108380595998e-04
            5.83538087981567143991412649484118e-02 1.85584469043908713639656760108210e-03 1.60466588787147433235388827021950e-04
            5.83538087981567143991412649484118e-02 9.39790346511404139917544853233267e-01 1.60466588787147433235388827021950e-04
            1.85584469043908713639656760108210e-03 5.83538087981567143991412649484118e-02 1.60466588787147433235388827021950e-04
            1.85584469043908713639656760108210e-03 9.39790346511404139917544853233267e-01 1.60466588787147433235388827021950e-04
            9.39790346511404139917544853233267e-01 5.83538087981567143991412649484118e-02 1.60466588787147433235388827021950e-04
            9.39790346511404139917544853233267e-01 1.85584469043908713639656760108210e-03 1.60466588787147433235388827021950e-04
            3.53736217391187435232779989746632e-01 1.51861221499153403735604683788551e-01 2.49050430530871769879697197325186e-03
            3.53736217391187435232779989746632e-01 4.94402561109659188787190942093730e-01 2.49050430530871769879697197325186e-03
            1.51861221499153403735604683788551e-01 3.53736217391187435232779989746632e-01 2.49050430530871769879697197325186e-03
            1.51861221499153403735604683788551e-01 4.94402561109659188787190942093730e-01 2.49050430530871769879697197325186e-03
            4.94402561109659188787190942093730e-01 3.53736217391187435232779989746632e-01 2.49050430530871769879697197325186e-03
            4.94402561109659188787190942093730e-01 1.51861221499153403735604683788551e-01 2.49050430530871769879697197325186e-03
            1.94103683421922135199366721280967e-01 1.46693846495503765758883218950359e-01 2.50978895904076194575682556830998e-03
            1.94103683421922135199366721280967e-01 6.59202470082574154552901291026501e-01 2.50978895904076194575682556830998e-03
            1.46693846495503765758883218950359e-01 1.94103683421922135199366721280967e-01 2.50978895904076194575682556830998e-03
            1.46693846495503765758883218950359e-01 6.59202470082574154552901291026501e-01 2.50978895904076194575682556830998e-03
            6.59202470082574154552901291026501e-01 1.94103683421922135199366721280967e-01 2.50978895904076194575682556830998e-03
            6.59202470082574154552901291026501e-01 1.46693846495503765758883218950359e-01 2.50978895904076194575682556830998e-03
            1.15244680262074267185212761432922e-01 7.30582724234847963318273400545877e-02 1.38461640914676353299184174971970e-03
            1.15244680262074267185212761432922e-01 8.11697047314440922605172090698034e-01 1.38461640914676353299184174971970e-03
            7.30582724234847963318273400545877e-02 1.15244680262074267185212761432922e-01 1.38461640914676353299184174971970e-03
            7.30582724234847963318273400545877e-02 8.11697047314440922605172090698034e-01 1.38461640914676353299184174971970e-03
            8.11697047314440922605172090698034e-01 1.15244680262074267185212761432922e-01 1.38461640914676353299184174971970e-03
            8.11697047314440922605172090698034e-01 7.30582724234847963318273400545877e-02 1.38461640914676353299184174971970e-03
            2.52648918856827386925090195290977e-01 8.33205145800466895344982276583323e-02 1.85079032983941375520298144863318e-03
            2.52648918856827386925090195290977e-01 6.64030566563125868029260345792864e-01 1.85079032983941375520298144863318e-03
            8.33205145800466895344982276583323e-02 2.52648918856827386925090195290977e-01 1.85079032983941375520298144863318e-03
            8.33205145800466895344982276583323e-02 6.64030566563125868029260345792864e-01 1.85079032983941375520298144863318e-03
            6.64030566563125868029260345792864e-01 2.52648918856827386925090195290977e-01 1.85079032983941375520298144863318e-03
            6.64030566563125868029260345792864e-01 8.33205145800466895344982276583323e-02 1.85079032983941375520298144863318e-03
            3.70638127911035064609990286044194e-01 9.76700995193537646610337787933531e-02 2.53751872958099793792241527512488e-03
            3.70638127911035064609990286044194e-01 5.31691772569611087462249088275712e-01 2.53751872958099793792241527512488e-03
            9.76700995193537646610337787933531e-02 3.70638127911035064609990286044194e-01 2.53751872958099793792241527512488e-03
            9.76700995193537646610337787933531e-02 5.31691772569611087462249088275712e-01 2.53751872958099793792241527512488e-03
            5.31691772569611087462249088275712e-01 3.70638127911035064609990286044194e-01 2.53751872958099793792241527512488e-03
            5.31691772569611087462249088275712e-01 9.76700995193537646610337787933531e-02 2.53751872958099793792241527512488e-03
            1.97510766235328455442399331332126e-01 1.02127473235820234731541233941243e-01 2.13383834577409023033900403731877e-03
            1.97510766235328455442399331332126e-01 7.00361760528851240437120395654347e-01 2.13383834577409023033900403731877e-03
            1.02127473235820234731541233941243e-01 1.97510766235328455442399331332126e-01 2.13383834577409023033900403731877e-03
            1.02127473235820234731541233941243e-01 7.00361760528851240437120395654347e-01 2.13383834577409023033900403731877e-03
            7.00361760528851240437120395654347e-01 1.97510766235328455442399331332126e-01 2.13383834577409023033900403731877e-03
            7.00361760528851240437120395654347e-01 1.02127473235820234731541233941243e-01 2.13383834577409023033900403731877e-03
            2.19102178776639594515174280786596e-01 1.69401226467935060726599516556234e-02 8.64613651994067212644412556699081e-04
            2.19102178776639594515174280786596e-01 7.63957698576566923698294431233080e-01 8.64613651994067212644412556699081e-04
            1.69401226467935060726599516556234e-02 2.19102178776639594515174280786596e-01 8.64613651994067212644412556699081e-04
            1.69401226467935060726599516556234e-02 7.63957698576566923698294431233080e-01 8.64613651994067212644412556699081e-04
            7.63957698576566923698294431233080e-01 2.19102178776639594515174280786596e-01 8.64613651994067212644412556699081e-04
            7.63957698576566923698294431233080e-01 1.69401226467935060726599516556234e-02 8.64613651994067212644412556699081e-04
            1.64749471455534146624444247208885e-01 6.46074146543942490206191564539040e-02 1.61236995612197434382673044694911e-03
            1.64749471455534146624444247208885e-01 7.70643113890071562721573172893841e-01 1.61236995612197434382673044694911e-03
            6.46074146543942490206191564539040e-02 1.64749471455534146624444247208885e-01 1.61236995612197434382673044694911e-03
            6.46074146543942490206191564539040e-02 7.70643113890071562721573172893841e-01 1.61236995612197434382673044694911e-03
            7.70643113890071562721573172893841e-01 1.64749471455534146624444247208885e-01 1.61236995612197434382673044694911e-03
            7.70643113890071562721573172893841e-01 6.46074146543942490206191564539040e-02 1.61236995612197434382673044694911e-03
            3.48756887430614948009122144867433e-01 1.94293491861925404773359105092823e-01 2.64397385648094821256059994141197e-03
            3.48756887430614948009122144867433e-01 4.56949620707459647217518750039744e-01 2.64397385648094821256059994141197e-03
            1.94293491861925404773359105092823e-01 3.48756887430614948009122144867433e-01 2.64397385648094821256059994141197e-03
            1.94293491861925404773359105092823e-01 4.56949620707459647217518750039744e-01 2.64397385648094821256059994141197e-03
            4.56949620707459647217518750039744e-01 3.48756887430614948009122144867433e-01 2.64397385648094821256059994141197e-03
            4.56949620707459647217518750039744e-01 1.94293491861925404773359105092823e-01 2.64397385648094821256059994141197e-03
            3.55853330058662320478646279298118e-01 5.87895877753212212080136112035689e-02 2.00979214318253510129785865956364e-03
            3.55853330058662320478646279298118e-01 5.85357082166016451374446205591084e-01 2.00979214318253510129785865956364e-03
            5.87895877753212212080136112035689e-02 3.55853330058662320478646279298118e-01 2.00979214318253510129785865956364e-03
            5.87895877753212212080136112035689e-02 5.85357082166016451374446205591084e-01 2.00979214318253510129785865956364e-03
            5.85357082166016451374446205591084e-01 3.55853330058662320478646279298118e-01 2.00979214318253510129785865956364e-03
            5.85357082166016451374446205591084e-01 5.87895877753212212080136112035689e-02 2.00979214318253510129785865956364e-03
            3.48716528985469398360663717539865e-01 2.43584975885173138365757949941326e-01 3.03041660365002338273221482722874e-03
            3.48716528985469398360663717539865e-01 4.07698495129357407762427101260982e-01 3.03041660365002338273221482722874e-03
            2.43584975885173138365757949941326e-01 3.48716528985469398360663717539865e-01 3.03041660365002338273221482722874e-03
            2.43584975885173138365757949941326e-01 4.07698495129357407762427101260982e-01 3.03041660365002338273221482722874e-03
            4.07698495129357407762427101260982e-01 3.48716528985469398360663717539865e-01 3.03041660365002338273221482722874e-03
            4.07698495129357407762427101260982e-01 2.43584975885173138365757949941326e-01 3.03041660365002338273221482722874e-03
            2.90704335549696046836487539621885e-01 1.86125965956163291759040134820680e-01 2.89956852928164102167762905537529e-03
            2.90704335549696046836487539621885e-01 5.23169698494140633648896709928522e-01 2.89956852928164102167762905537529e-03
            1.86125965956163291759040134820680e-01 2.90704335549696046836487539621885e-01 2.89956852928164102167762905537529e-03
            1.86125965956163291759040134820680e-01 5.23169698494140633648896709928522e-01 2.89956852928164102167762905537529e-03
            5.23169698494140633648896709928522e-01 2.90704335549696046836487539621885e-01 2.89956852928164102167762905537529e-03
            5.23169698494140633648896709928522e-01 1.86125965956163291759040134820680e-01 2.89956852928164102167762905537529e-03
            2.89862934817113671037702715693740e-01 2.38357019552556226305028985734680e-01 3.19657376055502030071098218400039e-03
            2.89862934817113671037702715693740e-01 4.71780045630330102657268298571580e-01 3.19657376055502030071098218400039e-03
            2.38357019552556226305028985734680e-01 2.89862934817113671037702715693740e-01 3.19657376055502030071098218400039e-03
            2.38357019552556226305028985734680e-01 4.71780045630330102657268298571580e-01 3.19657376055502030071098218400039e-03
            4.71780045630330102657268298571580e-01 2.89862934817113671037702715693740e-01 3.19657376055502030071098218400039e-03
            4.71780045630330102657268298571580e-01 2.38357019552556226305028985734680e-01 3.19657376055502030071098218400039e-03
            3.09383575445457970065632480327622e-01 1.31229616791972980571756579593057e-01 2.82733750449421557954243766630498e-03
            3.09383575445457970065632480327622e-01 5.59386807762569104873762171337148e-01 2.82733750449421557954243766630498e-03
            1.31229616791972980571756579593057e-01 3.09383575445457970065632480327622e-01 2.82733750449421557954243766630498e-03
            1.31229616791972980571756579593057e-01 5.59386807762569104873762171337148e-01 2.82733750449421557954243766630498e-03
            5.59386807762569104873762171337148e-01 3.09383575445457970065632480327622e-01 2.82733750449421557954243766630498e-03
            5.59386807762569104873762171337148e-01 1.31229616791972980571756579593057e-01 2.82733750449421557954243766630498e-03
            2.46979539099244443933756087972142e-01 1.68458978552479210488712624282925e-01 2.65552470844998469295950016544339e-03
            2.46979539099244443933756087972142e-01 5.84561482348276317821955672116019e-01 2.65552470844998469295950016544339e-03
            1.68458978552479210488712624282925e-01 2.46979539099244443933756087972142e-01 2.65552470844998469295950016544339e-03
            1.68458978552479210488712624282925e-01 5.84561482348276317821955672116019e-01 2.65552470844998469295950016544339e-03
            5.84561482348276317821955672116019e-01 2.46979539099244443933756087972142e-01 2.65552470844998469295950016544339e-03
            5.84561482348276317821955672116019e-01 1.68458978552479210488712624282925e-01 2.65552470844998469295950016544339e-03
            4.35345195121465189380671745311702e-01 1.31254695112234757298219633980807e-02 9.86615210737949374641964084275969e-04
            4.35345195121465189380671745311702e-01 5.51529335367311301929760247730883e-01 9.86615210737949374641964084275969e-04
            1.31254695112234757298219633980807e-02 4.35345195121465189380671745311702e-01 9.86615210737949374641964084275969e-04
            1.31254695112234757298219633980807e-02 5.51529335367311301929760247730883e-01 9.86615210737949374641964084275969e-04
            5.51529335367311301929760247730883e-01 4.35345195121465189380671745311702e-01 9.86615210737949374641964084275969e-04
            5.51529335367311301929760247730883e-01 1.31254695112234757298219633980807e-02 9.86615210737949374641964084275969e-04
            4.13701289314883091652319535569404e-01 3.44134405185802239035552929635742e-02 1.77773973578346949388218067866774e-03
            4.13701289314883091652319535569404e-01 5.51885270166536767710852018353762e-01 1.77773973578346949388218067866774e-03
            3.44134405185802239035552929635742e-02 4.13701289314883091652319535569404e-01 1.77773973578346949388218067866774e-03
            3.44134405185802239035552929635742e-02 5.51885270166536767710852018353762e-01 1.77773973578346949388218067866774e-03
            5.51885270166536767710852018353762e-01 4.13701289314883091652319535569404e-01 1.77773973578346949388218067866774e-03
            5.51885270166536767710852018353762e-01 3.44134405185802239035552929635742e-02 1.77773973578346949388218067866774e-03
            ];
        
    case  45
        
        % ALG. DEG.:   45
        % PTS CARD.:  370
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.5e-15
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.47073482719251202865962113719434e-03
            1.29806653300276658713485744556237e-01 1.29806653300276658713485744556237e-01 1.60120857517628638853890432613980e-03
            1.29806653300276658713485744556237e-01 7.40386693399446738084179742145352e-01 1.60120857517628638853890432613980e-03
            7.40386693399446738084179742145352e-01 1.29806653300276658713485744556237e-01 1.60120857517628638853890432613980e-03
            2.06349665630883027933606399528799e-01 2.06349665630883027933606399528799e-01 2.31867442056098637878647217291928e-03
            2.06349665630883027933606399528799e-01 5.87300668738233944132787200942403e-01 2.31867442056098637878647217291928e-03
            5.87300668738233944132787200942403e-01 2.06349665630883027933606399528799e-01 2.31867442056098637878647217291928e-03
            1.75084588806939833727227551207761e-01 1.75084588806939833727227551207761e-01 2.04353023274364813816061925422218e-03
            1.75084588806939833727227551207761e-01 6.49830822386120332545544897584477e-01 2.04353023274364813816061925422218e-03
            6.49830822386120332545544897584477e-01 1.75084588806939833727227551207761e-01 2.04353023274364813816061925422218e-03
            4.99504576867959404484764718290535e-01 4.99504576867959404484764718290535e-01 2.41556761516012848600362672435438e-04
            4.99504576867959404484764718290535e-01 9.90846264081191030470563418930396e-04 2.41556761516012848600362672435438e-04
            9.90846264081191030470563418930396e-04 4.99504576867959404484764718290535e-01 2.41556761516012848600362672435438e-04
            5.52204190605027900318191313999705e-02 5.52204190605027900318191313999705e-02 8.46305070277619114760270235819917e-04
            5.52204190605027900318191313999705e-02 8.89559161878994419936361737200059e-01 8.46305070277619114760270235819917e-04
            8.89559161878994419936361737200059e-01 5.52204190605027900318191313999705e-02 8.46305070277619114760270235819917e-04
            2.99152765836593892423422857973492e-01 2.99152765836593892423422857973492e-01 3.41687768784906158517356011827815e-03
            2.99152765836593892423422857973492e-01 4.01694468326812215153154284053016e-01 3.41687768784906158517356011827815e-03
            4.01694468326812215153154284053016e-01 2.99152765836593892423422857973492e-01 3.41687768784906158517356011827815e-03
            3.66982277256458211400769187093829e-01 3.66982277256458211400769187093829e-01 3.44625239929347623041300252566543e-03
            3.66982277256458211400769187093829e-01 2.66035445487083577198461625812342e-01 3.44625239929347623041300252566543e-03
            2.66035445487083577198461625812342e-01 3.66982277256458211400769187093829e-01 3.44625239929347623041300252566543e-03
            2.64921642390380518428116829454666e-01 2.64921642390380518428116829454666e-01 3.15732849651644865984767562849811e-03
            2.64921642390380518428116829454666e-01 4.70156715219238963143766341090668e-01 3.15732849651644865984767562849811e-03
            4.70156715219238963143766341090668e-01 2.64921642390380518428116829454666e-01 3.15732849651644865984767562849811e-03
            3.98469068413945026740918819996295e-01 3.98469068413945026740918819996295e-01 3.27915375009894208974126073030675e-03
            3.98469068413945026740918819996295e-01 2.03061863172109946518162360007409e-01 3.27915375009894208974126073030675e-03
            2.03061863172109946518162360007409e-01 3.98469068413945026740918819996295e-01 3.27915375009894208974126073030675e-03
            4.52620190751494178016400837805122e-01 4.52620190751494178016400837805122e-01 2.50842974298410518324553564184498e-03
            4.52620190751494178016400837805122e-01 9.47596184970116439671983243897557e-02 2.50842974298410518324553564184498e-03
            9.47596184970116439671983243897557e-02 4.52620190751494178016400837805122e-01 2.50842974298410518324553564184498e-03
            4.26861049533307446157692766064429e-01 4.26861049533307446157692766064429e-01 2.94125456308764977528258377503789e-03
            4.26861049533307446157692766064429e-01 1.46277900933385107684614467871143e-01 2.94125456308764977528258377503789e-03
            1.46277900933385107684614467871143e-01 4.26861049533307446157692766064429e-01 2.94125456308764977528258377503789e-03
            9.45724371334943636480119266707334e-02 9.45724371334943636480119266707334e-02 1.56857173108213422511869605813217e-03
            9.45724371334943636480119266707334e-02 8.10855125733011217192824915400706e-01 1.56857173108213422511869605813217e-03
            8.10855125733011217192824915400706e-01 9.45724371334943636480119266707334e-02 1.56857173108213422511869605813217e-03
            4.72989470868897310396050670533441e-01 4.72989470868897310396050670533441e-01 2.04653257390983479846457981921048e-03
            4.72989470868897310396050670533441e-01 5.40210582622053792078986589331180e-02 2.04653257390983479846457981921048e-03
            5.40210582622053792078986589331180e-02 4.72989470868897310396050670533441e-01 2.04653257390983479846457981921048e-03
            3.40851411567991424700352354193456e-02 3.40851411567991424700352354193456e-02 6.25657342660229350353451316379960e-04
            3.40851411567991424700352354193456e-02 9.31829717686401659548778297903482e-01 6.25657342660229350353451316379960e-04
            9.31829717686401659548778297903482e-01 3.40851411567991424700352354193456e-02 6.25657342660229350353451316379960e-04
            4.86998996825781460540127909553121e-01 4.86998996825781460540127909553121e-01 1.52790288514941473806130201040787e-03
            4.86998996825781460540127909553121e-01 2.60020063484370789197441808937583e-02 1.52790288514941473806130201040787e-03
            2.60020063484370789197441808937583e-02 4.86998996825781460540127909553121e-01 1.52790288514941473806130201040787e-03
            1.26048038307361616461310660497475e-02 7.29242955084338090321338654575811e-03 9.71532123877305272326848628594576e-05
            1.26048038307361616461310660497475e-02 9.80102766618420440103420787636423e-01 9.71532123877305272326848628594576e-05
            7.29242955084338090321338654575811e-03 1.26048038307361616461310660497475e-02 9.71532123877305272326848628594576e-05
            7.29242955084338090321338654575811e-03 9.80102766618420440103420787636423e-01 9.71532123877305272326848628594576e-05
            9.80102766618420440103420787636423e-01 1.26048038307361616461310660497475e-02 9.71532123877305272326848628594576e-05
            9.80102766618420440103420787636423e-01 7.29242955084338090321338654575811e-03 9.71532123877305272326848628594576e-05
            4.60547663144127931295068734129927e-03 1.34999718211049760455133572634168e-03 2.96832401489165982686867922835106e-05
            4.60547663144127931295068734129927e-03 9.94044526186448229587711011845386e-01 2.96832401489165982686867922835106e-05
            1.34999718211049760455133572634168e-03 4.60547663144127931295068734129927e-03 2.96832401489165982686867922835106e-05
            1.34999718211049760455133572634168e-03 9.94044526186448229587711011845386e-01 2.96832401489165982686867922835106e-05
            9.94044526186448229587711011845386e-01 4.60547663144127931295068734129927e-03 2.96832401489165982686867922835106e-05
            9.94044526186448229587711011845386e-01 1.34999718211049760455133572634168e-03 2.96832401489165982686867922835106e-05
            6.79464966039111523121718505535682e-02 6.85917593806053670774103281360112e-03 2.40344353751331478961927867743498e-04
            6.79464966039111523121718505535682e-02 9.25194327458028320521066234505270e-01 2.40344353751331478961927867743498e-04
            6.85917593806053670774103281360112e-03 6.79464966039111523121718505535682e-02 2.40344353751331478961927867743498e-04
            6.85917593806053670774103281360112e-03 9.25194327458028320521066234505270e-01 2.40344353751331478961927867743498e-04
            9.25194327458028320521066234505270e-01 6.79464966039111523121718505535682e-02 2.40344353751331478961927867743498e-04
            9.25194327458028320521066234505270e-01 6.85917593806053670774103281360112e-03 2.40344353751331478961927867743498e-04
            6.93380043634408277375413831578044e-02 1.22976799593489888665076570362089e-03 1.15062466750111980088103202923833e-04
            6.93380043634408277375413831578044e-02 9.29432227640624275544212196109584e-01 1.15062466750111980088103202923833e-04
            1.22976799593489888665076570362089e-03 6.93380043634408277375413831578044e-02 1.15062466750111980088103202923833e-04
            1.22976799593489888665076570362089e-03 9.29432227640624275544212196109584e-01 1.15062466750111980088103202923833e-04
            9.29432227640624275544212196109584e-01 6.93380043634408277375413831578044e-02 1.15062466750111980088103202923833e-04
            9.29432227640624275544212196109584e-01 1.22976799593489888665076570362089e-03 1.15062466750111980088103202923833e-04
            4.68270859870947964154197507014032e-01 8.78010020069475149995241736178286e-03 6.47231358534071528863795830233130e-04
            4.68270859870947964154197507014032e-01 5.22949039928357284345850075624185e-01 6.47231358534071528863795830233130e-04
            8.78010020069475149995241736178286e-03 4.68270859870947964154197507014032e-01 6.47231358534071528863795830233130e-04
            8.78010020069475149995241736178286e-03 5.22949039928357284345850075624185e-01 6.47231358534071528863795830233130e-04
            5.22949039928357284345850075624185e-01 4.68270859870947964154197507014032e-01 6.47231358534071528863795830233130e-04
            5.22949039928357284345850075624185e-01 8.78010020069475149995241736178286e-03 6.47231358534071528863795830233130e-04
            2.50495170335221506441314431867795e-01 1.68302636118027710253330919698556e-03 2.30260449145342833325925124832168e-04
            2.50495170335221506441314431867795e-01 7.47821803303598264811569151788717e-01 2.30260449145342833325925124832168e-04
            1.68302636118027710253330919698556e-03 2.50495170335221506441314431867795e-01 2.30260449145342833325925124832168e-04
            1.68302636118027710253330919698556e-03 7.47821803303598264811569151788717e-01 2.30260449145342833325925124832168e-04
            7.47821803303598264811569151788717e-01 2.50495170335221506441314431867795e-01 2.30260449145342833325925124832168e-04
            7.47821803303598264811569151788717e-01 1.68302636118027710253330919698556e-03 2.30260449145342833325925124832168e-04
            1.79231013398374397493917342671921e-02 1.75797258353514685921825666525820e-03 7.75345040014330428874284040219322e-05
            1.79231013398374397493917342671921e-02 9.80318926076627406018815236166120e-01 7.75345040014330428874284040219322e-05
            1.75797258353514685921825666525820e-03 1.79231013398374397493917342671921e-02 7.75345040014330428874284040219322e-05
            1.75797258353514685921825666525820e-03 9.80318926076627406018815236166120e-01 7.75345040014330428874284040219322e-05
            9.80318926076627406018815236166120e-01 1.79231013398374397493917342671921e-02 7.75345040014330428874284040219322e-05
            9.80318926076627406018815236166120e-01 1.75797258353514685921825666525820e-03 7.75345040014330428874284040219322e-05
            3.59456841995052789329179177002516e-01 1.47561365316294709065791934676781e-03 2.63216008449837382630193127752705e-04
            3.59456841995052789329179177002516e-01 6.39067544351784255773907261755085e-01 2.63216008449837382630193127752705e-04
            1.47561365316294709065791934676781e-03 3.59456841995052789329179177002516e-01 2.63216008449837382630193127752705e-04
            1.47561365316294709065791934676781e-03 6.39067544351784255773907261755085e-01 2.63216008449837382630193127752705e-04
            6.39067544351784255773907261755085e-01 3.59456841995052789329179177002516e-01 2.63216008449837382630193127752705e-04
            6.39067544351784255773907261755085e-01 1.47561365316294709065791934676781e-03 2.63216008449837382630193127752705e-04
            2.21153984784947493791307238097943e-01 1.53663127462592530303453486340004e-01 2.03880140043859832063577464111859e-03
            2.21153984784947493791307238097943e-01 6.25182887752459892638512428675313e-01 2.03880140043859832063577464111859e-03
            1.53663127462592530303453486340004e-01 2.21153984784947493791307238097943e-01 2.03880140043859832063577464111859e-03
            1.53663127462592530303453486340004e-01 6.25182887752459892638512428675313e-01 2.03880140043859832063577464111859e-03
            6.25182887752459892638512428675313e-01 2.21153984784947493791307238097943e-01 2.03880140043859832063577464111859e-03
            6.25182887752459892638512428675313e-01 1.53663127462592530303453486340004e-01 2.03880140043859832063577464111859e-03
            2.38973194878696115761584906067583e-01 2.27625216137249258840924426294805e-01 1.36366857778165797600300468417345e-03
            2.38973194878696115761584906067583e-01 5.33401588984054542130763820750872e-01 1.36366857778165797600300468417345e-03
            2.27625216137249258840924426294805e-01 2.38973194878696115761584906067583e-01 1.36366857778165797600300468417345e-03
            2.27625216137249258840924426294805e-01 5.33401588984054542130763820750872e-01 1.36366857778165797600300468417345e-03
            5.33401588984054542130763820750872e-01 2.38973194878696115761584906067583e-01 1.36366857778165797600300468417345e-03
            5.33401588984054542130763820750872e-01 2.27625216137249258840924426294805e-01 1.36366857778165797600300468417345e-03
            2.26522885981175758907113504392328e-01 1.14203037751511859987374464253662e-01 1.90891918002090770975498479344878e-03
            2.26522885981175758907113504392328e-01 6.59274076267312381105512031354010e-01 1.90891918002090770975498479344878e-03
            1.14203037751511859987374464253662e-01 2.26522885981175758907113504392328e-01 1.90891918002090770975498479344878e-03
            1.14203037751511859987374464253662e-01 6.59274076267312381105512031354010e-01 1.90891918002090770975498479344878e-03
            6.59274076267312381105512031354010e-01 2.26522885981175758907113504392328e-01 1.90891918002090770975498479344878e-03
            6.59274076267312381105512031354010e-01 1.14203037751511859987374464253662e-01 1.90891918002090770975498479344878e-03
            2.39590524037790186806518022422097e-01 8.03750594685513908288498896581586e-02 1.62179496424432779966284812189770e-03
            2.39590524037790186806518022422097e-01 6.80034416493658366853480856661918e-01 1.62179496424432779966284812189770e-03
            8.03750594685513908288498896581586e-02 2.39590524037790186806518022422097e-01 1.62179496424432779966284812189770e-03
            8.03750594685513908288498896581586e-02 6.80034416493658366853480856661918e-01 1.62179496424432779966284812189770e-03
            6.80034416493658366853480856661918e-01 2.39590524037790186806518022422097e-01 1.62179496424432779966284812189770e-03
            6.80034416493658366853480856661918e-01 8.03750594685513908288498896581586e-02 1.62179496424432779966284812189770e-03
            1.78291402992592273868410757131642e-01 3.06183636824900254347348038663768e-02 9.42482859192453098669262079312148e-04
            1.78291402992592273868410757131642e-01 7.91090233324917679880172727280296e-01 9.42482859192453098669262079312148e-04
            3.06183636824900254347348038663768e-02 1.78291402992592273868410757131642e-01 9.42482859192453098669262079312148e-04
            3.06183636824900254347348038663768e-02 7.91090233324917679880172727280296e-01 9.42482859192453098669262079312148e-04
            7.91090233324917679880172727280296e-01 1.78291402992592273868410757131642e-01 9.42482859192453098669262079312148e-04
            7.91090233324917679880172727280296e-01 3.06183636824900254347348038663768e-02 9.42482859192453098669262079312148e-04
            2.82586689120417511311700309306616e-01 2.69421027814230988617349993319294e-02 1.09036119380348994409302143537843e-03
            2.82586689120417511311700309306616e-01 6.90471208098159427990481162851211e-01 1.09036119380348994409302143537843e-03
            2.69421027814230988617349993319294e-02 2.82586689120417511311700309306616e-01 1.09036119380348994409302143537843e-03
            2.69421027814230988617349993319294e-02 6.90471208098159427990481162851211e-01 1.09036119380348994409302143537843e-03
            6.90471208098159427990481162851211e-01 2.82586689120417511311700309306616e-01 1.09036119380348994409302143537843e-03
            6.90471208098159427990481162851211e-01 2.69421027814230988617349993319294e-02 1.09036119380348994409302143537843e-03
            1.37257110258729553287082580936840e-01 3.27972135469326506185261393966357e-02 9.17290619580521512585469245948389e-04
            1.37257110258729553287082580936840e-01 8.29945676194337789155497375759296e-01 9.17290619580521512585469245948389e-04
            3.27972135469326506185261393966357e-02 1.37257110258729553287082580936840e-01 9.17290619580521512585469245948389e-04
            3.27972135469326506185261393966357e-02 8.29945676194337789155497375759296e-01 9.17290619580521512585469245948389e-04
            8.29945676194337789155497375759296e-01 1.37257110258729553287082580936840e-01 9.17290619580521512585469245948389e-04
            8.29945676194337789155497375759296e-01 3.27972135469326506185261393966357e-02 9.17290619580521512585469245948389e-04
            1.00254415236569610625494419764436e-01 3.27271127336489636716621021150786e-02 8.13669897716354098907154401842945e-04
            1.00254415236569610625494419764436e-01 8.67018472029781439580631285934942e-01 8.13669897716354098907154401842945e-04
            3.27271127336489636716621021150786e-02 1.00254415236569610625494419764436e-01 8.13669897716354098907154401842945e-04
            3.27271127336489636716621021150786e-02 8.67018472029781439580631285934942e-01 8.13669897716354098907154401842945e-04
            8.67018472029781439580631285934942e-01 1.00254415236569610625494419764436e-01 8.13669897716354098907154401842945e-04
            8.67018472029781439580631285934942e-01 3.27271127336489636716621021150786e-02 8.13669897716354098907154401842945e-04
            5.17074765599652777514627643995482e-02 1.51601013138725621010038935310149e-02 4.14266410586361995705556582336726e-04
            5.17074765599652777514627643995482e-02 9.33132422126162097697488206904382e-01 4.14266410586361995705556582336726e-04
            1.51601013138725621010038935310149e-02 5.17074765599652777514627643995482e-02 4.14266410586361995705556582336726e-04
            1.51601013138725621010038935310149e-02 9.33132422126162097697488206904382e-01 4.14266410586361995705556582336726e-04
            9.33132422126162097697488206904382e-01 5.17074765599652777514627643995482e-02 4.14266410586361995705556582336726e-04
            9.33132422126162097697488206904382e-01 1.51601013138725621010038935310149e-02 4.14266410586361995705556582336726e-04
            2.60614855137656920014421757514356e-01 1.70884652952977555973035350689315e-01 2.55374504338384258547711702647121e-03
            2.60614855137656920014421757514356e-01 5.68500491909365468501391660538502e-01 2.55374504338384258547711702647121e-03
            1.70884652952977555973035350689315e-01 2.60614855137656920014421757514356e-01 2.55374504338384258547711702647121e-03
            1.70884652952977555973035350689315e-01 5.68500491909365468501391660538502e-01 2.55374504338384258547711702647121e-03
            5.68500491909365468501391660538502e-01 2.60614855137656920014421757514356e-01 2.55374504338384258547711702647121e-03
            5.68500491909365468501391660538502e-01 1.70884652952977555973035350689315e-01 2.55374504338384258547711702647121e-03
            2.27214324959957303384783244837308e-01 2.86632862041672348274445880633721e-02 1.11955667944577640093850590119473e-03
            2.27214324959957303384783244837308e-01 7.44122388835875447909984359284863e-01 1.11955667944577640093850590119473e-03
            2.86632862041672348274445880633721e-02 2.27214324959957303384783244837308e-01 1.11955667944577640093850590119473e-03
            2.86632862041672348274445880633721e-02 7.44122388835875447909984359284863e-01 1.11955667944577640093850590119473e-03
            7.44122388835875447909984359284863e-01 2.27214324959957303384783244837308e-01 1.11955667944577640093850590119473e-03
            7.44122388835875447909984359284863e-01 2.86632862041672348274445880633721e-02 1.11955667944577640093850590119473e-03
            3.11040487235050899794686074528727e-01 1.26956715659609486318570858998100e-02 8.20086182210518952438726980602723e-04
            3.11040487235050899794686074528727e-01 6.76263841198988080449794324522372e-01 8.20086182210518952438726980602723e-04
            1.26956715659609486318570858998100e-02 3.11040487235050899794686074528727e-01 8.20086182210518952438726980602723e-04
            1.26956715659609486318570858998100e-02 6.76263841198988080449794324522372e-01 8.20086182210518952438726980602723e-04
            6.76263841198988080449794324522372e-01 3.11040487235050899794686074528727e-01 8.20086182210518952438726980602723e-04
            6.76263841198988080449794324522372e-01 1.26956715659609486318570858998100e-02 8.20086182210518952438726980602723e-04
            1.85623876299053219840473616386589e-01 8.79015352488810713937894547598262e-02 1.83583569658886782208218058087823e-03
            1.85623876299053219840473616386589e-01 7.26474588452065694887949121039128e-01 1.83583569658886782208218058087823e-03
            8.79015352488810713937894547598262e-02 1.85623876299053219840473616386589e-01 1.83583569658886782208218058087823e-03
            8.79015352488810713937894547598262e-02 7.26474588452065694887949121039128e-01 1.83583569658886782208218058087823e-03
            7.26474588452065694887949121039128e-01 1.85623876299053219840473616386589e-01 1.83583569658886782208218058087823e-03
            7.26474588452065694887949121039128e-01 8.79015352488810713937894547598262e-02 1.83583569658886782208218058087823e-03
            6.58626437608177683902255239445367e-02 3.07203461767424183814778615442265e-02 7.06502470799715205120039573927215e-04
            6.58626437608177683902255239445367e-02 9.03417010062439795881061854743166e-01 7.06502470799715205120039573927215e-04
            3.07203461767424183814778615442265e-02 6.58626437608177683902255239445367e-02 7.06502470799715205120039573927215e-04
            3.07203461767424183814778615442265e-02 9.03417010062439795881061854743166e-01 7.06502470799715205120039573927215e-04
            9.03417010062439795881061854743166e-01 6.58626437608177683902255239445367e-02 7.06502470799715205120039573927215e-04
            9.03417010062439795881061854743166e-01 3.07203461767424183814778615442265e-02 7.06502470799715205120039573927215e-04
            3.01590470095362273017514098683023e-01 2.81455063987638438166571930310056e-03 3.82221000431039019135798229243051e-04
            3.01590470095362273017514098683023e-01 6.95594979264761437143249622749863e-01 3.82221000431039019135798229243051e-04
            2.81455063987638438166571930310056e-03 3.01590470095362273017514098683023e-01 3.82221000431039019135798229243051e-04
            2.81455063987638438166571930310056e-03 6.95594979264761437143249622749863e-01 3.82221000431039019135798229243051e-04
            6.95594979264761437143249622749863e-01 3.01590470095362273017514098683023e-01 3.82221000431039019135798229243051e-04
            6.95594979264761437143249622749863e-01 2.81455063987638438166571930310056e-03 3.82221000431039019135798229243051e-04
            2.92542657654236315512008559380774e-01 7.98394163722915967129623027176422e-02 1.97701581302659997255166679508420e-03
            2.92542657654236315512008559380774e-01 6.27617925973472101652816945716040e-01 1.97701581302659997255166679508420e-03
            7.98394163722915967129623027176422e-02 2.92542657654236315512008559380774e-01 1.97701581302659997255166679508420e-03
            7.98394163722915967129623027176422e-02 6.27617925973472101652816945716040e-01 1.97701581302659997255166679508420e-03
            6.27617925973472101652816945716040e-01 2.92542657654236315512008559380774e-01 1.97701581302659997255166679508420e-03
            6.27617925973472101652816945716040e-01 7.98394163722915967129623027176422e-02 1.97701581302659997255166679508420e-03
            2.73024847528849125277705667258488e-02 1.33211979027671369862906303183081e-02 3.22416772478557639609875185016108e-04
            2.73024847528849125277705667258488e-02 9.59376317344348028548495221912162e-01 3.22416772478557639609875185016108e-04
            1.33211979027671369862906303183081e-02 2.73024847528849125277705667258488e-02 3.22416772478557639609875185016108e-04
            1.33211979027671369862906303183081e-02 9.59376317344348028548495221912162e-01 3.22416772478557639609875185016108e-04
            9.59376317344348028548495221912162e-01 2.73024847528849125277705667258488e-02 3.22416772478557639609875185016108e-04
            9.59376317344348028548495221912162e-01 1.33211979027671369862906303183081e-02 3.22416772478557639609875185016108e-04
            3.71571920577544234642886067376821e-01 9.60218601811555806657683120874935e-03 7.77378451269038206472572483107797e-04
            3.71571920577544234642886067376821e-01 6.18825893404340132697427634411724e-01 7.77378451269038206472572483107797e-04
            9.60218601811555806657683120874935e-03 3.71571920577544234642886067376821e-01 7.77378451269038206472572483107797e-04
            9.60218601811555806657683120874935e-03 6.18825893404340132697427634411724e-01 7.77378451269038206472572483107797e-04
            6.18825893404340132697427634411724e-01 3.71571920577544234642886067376821e-01 7.77378451269038206472572483107797e-04
            6.18825893404340132697427634411724e-01 9.60218601811555806657683120874935e-03 7.77378451269038206472572483107797e-04
            2.01007303993687103105258984214743e-01 2.50243738523270699627976121348638e-03 3.13239896240612636021949377607143e-04
            2.01007303993687103105258984214743e-01 7.96490258621080249312740306777414e-01 3.13239896240612636021949377607143e-04
            2.50243738523270699627976121348638e-03 2.01007303993687103105258984214743e-01 3.13239896240612636021949377607143e-04
            2.50243738523270699627976121348638e-03 7.96490258621080249312740306777414e-01 3.13239896240612636021949377607143e-04
            7.96490258621080249312740306777414e-01 2.01007303993687103105258984214743e-01 3.13239896240612636021949377607143e-04
            7.96490258621080249312740306777414e-01 2.50243738523270699627976121348638e-03 3.13239896240612636021949377607143e-04
            2.93183129372266426670989858394023e-01 2.01730828065996542486359999202250e-01 2.99590287508692950865785142866571e-03
            2.93183129372266426670989858394023e-01 5.05086042561737058598225758032640e-01 2.99590287508692950865785142866571e-03
            2.01730828065996542486359999202250e-01 2.93183129372266426670989858394023e-01 2.99590287508692950865785142866571e-03
            2.01730828065996542486359999202250e-01 5.05086042561737058598225758032640e-01 2.99590287508692950865785142866571e-03
            5.05086042561737058598225758032640e-01 2.93183129372266426670989858394023e-01 2.99590287508692950865785142866571e-03
            5.05086042561737058598225758032640e-01 2.01730828065996542486359999202250e-01 2.99590287508692950865785142866571e-03
            2.33331891103622568461872788248002e-01 5.26776823006486205613896345312241e-02 1.52920890728200538355441295834680e-03
            2.33331891103622568461872788248002e-01 7.13990426595728755465586345962947e-01 1.52920890728200538355441295834680e-03
            5.26776823006486205613896345312241e-02 2.33331891103622568461872788248002e-01 1.52920890728200538355441295834680e-03
            5.26776823006486205613896345312241e-02 7.13990426595728755465586345962947e-01 1.52920890728200538355441295834680e-03
            7.13990426595728755465586345962947e-01 2.33331891103622568461872788248002e-01 1.52920890728200538355441295834680e-03
            7.13990426595728755465586345962947e-01 5.26776823006486205613896345312241e-02 1.52920890728200538355441295834680e-03
            1.79299141897004327139697466009238e-01 5.57691461107685942932832290352962e-02 1.49820395207656323864620073749165e-03
            1.79299141897004327139697466009238e-01 7.64931711992227092444807112769922e-01 1.49820395207656323864620073749165e-03
            5.57691461107685942932832290352962e-02 1.79299141897004327139697466009238e-01 1.49820395207656323864620073749165e-03
            5.57691461107685942932832290352962e-02 7.64931711992227092444807112769922e-01 1.49820395207656323864620073749165e-03
            7.64931711992227092444807112769922e-01 1.79299141897004327139697466009238e-01 1.49820395207656323864620073749165e-03
            7.64931711992227092444807112769922e-01 5.57691461107685942932832290352962e-02 1.49820395207656323864620073749165e-03
            2.93435331129092979995931500525330e-01 4.82367013092171934007801326060871e-02 1.61668679165861322102881736384461e-03
            2.93435331129092979995931500525330e-01 6.58327967561689875175545694219181e-01 1.61668679165861322102881736384461e-03
            4.82367013092171934007801326060871e-02 2.93435331129092979995931500525330e-01 1.61668679165861322102881736384461e-03
            4.82367013092171934007801326060871e-02 6.58327967561689875175545694219181e-01 1.61668679165861322102881736384461e-03
            6.58327967561689875175545694219181e-01 2.93435331129092979995931500525330e-01 1.61668679165861322102881736384461e-03
            6.58327967561689875175545694219181e-01 4.82367013092171934007801326060871e-02 1.61668679165861322102881736384461e-03
            3.29840596275619235555609520815779e-01 2.33697583553122173460891985996568e-01 3.32318218248998454092402177195709e-03
            3.29840596275619235555609520815779e-01 4.36461820171258563227922877558740e-01 3.32318218248998454092402177195709e-03
            2.33697583553122173460891985996568e-01 3.29840596275619235555609520815779e-01 3.32318218248998454092402177195709e-03
            2.33697583553122173460891985996568e-01 4.36461820171258563227922877558740e-01 3.32318218248998454092402177195709e-03
            4.36461820171258563227922877558740e-01 3.29840596275619235555609520815779e-01 3.32318218248998454092402177195709e-03
            4.36461820171258563227922877558740e-01 2.33697583553122173460891985996568e-01 3.32318218248998454092402177195709e-03
            1.38175693611527594661936291231541e-01 9.26621506087761254155665824328025e-02 1.59745246645443027613819886312285e-03
            1.38175693611527594661936291231541e-01 7.69162155779696266044709318521200e-01 1.59745246645443027613819886312285e-03
            9.26621506087761254155665824328025e-02 1.38175693611527594661936291231541e-01 1.59745246645443027613819886312285e-03
            9.26621506087761254155665824328025e-02 7.69162155779696266044709318521200e-01 1.59745246645443027613819886312285e-03
            7.69162155779696266044709318521200e-01 1.38175693611527594661936291231541e-01 1.59745246645443027613819886312285e-03
            7.69162155779696266044709318521200e-01 9.26621506087761254155665824328025e-02 1.59745246645443027613819886312285e-03
            3.58512865776101086190408295806265e-01 1.73454768103072393126140582353401e-01 3.08099584148886947165824068406437e-03
            3.58512865776101086190408295806265e-01 4.68032366120826548439026737469248e-01 3.08099584148886947165824068406437e-03
            1.73454768103072393126140582353401e-01 3.58512865776101086190408295806265e-01 3.08099584148886947165824068406437e-03
            1.73454768103072393126140582353401e-01 4.68032366120826548439026737469248e-01 3.08099584148886947165824068406437e-03
            4.68032366120826548439026737469248e-01 3.58512865776101086190408295806265e-01 3.08099584148886947165824068406437e-03
            4.68032366120826548439026737469248e-01 1.73454768103072393126140582353401e-01 3.08099584148886947165824068406437e-03
            4.20337783466548764454984166150098e-01 1.78508936465828042805537023696161e-02 1.09190886564487473278650675467816e-03
            4.20337783466548764454984166150098e-01 5.61811322886868524939529834227869e-01 1.09190886564487473278650675467816e-03
            1.78508936465828042805537023696161e-02 4.20337783466548764454984166150098e-01 1.09190886564487473278650675467816e-03
            1.78508936465828042805537023696161e-02 5.61811322886868524939529834227869e-01 1.09190886564487473278650675467816e-03
            5.61811322886868524939529834227869e-01 4.20337783466548764454984166150098e-01 1.09190886564487473278650675467816e-03
            5.61811322886868524939529834227869e-01 1.78508936465828042805537023696161e-02 1.09190886564487473278650675467816e-03
            2.79051133003480611449731441098265e-01 1.19946692495847162529543084019679e-01 2.41911437679862570751532224733182e-03
            2.79051133003480611449731441098265e-01 6.01002174500672170509574243624229e-01 2.41911437679862570751532224733182e-03
            1.19946692495847162529543084019679e-01 2.79051133003480611449731441098265e-01 2.41911437679862570751532224733182e-03
            1.19946692495847162529543084019679e-01 6.01002174500672170509574243624229e-01 2.41911437679862570751532224733182e-03
            6.01002174500672170509574243624229e-01 2.79051133003480611449731441098265e-01 2.41911437679862570751532224733182e-03
            6.01002174500672170509574243624229e-01 1.19946692495847162529543084019679e-01 2.41911437679862570751532224733182e-03
            9.34626722493710621053608633701515e-02 1.39846172517608963864477189531499e-02 5.85471759754055626548407964548915e-04
            9.34626722493710621053608633701515e-02 8.92552710498867973853975854581222e-01 5.85471759754055626548407964548915e-04
            1.39846172517608963864477189531499e-02 9.34626722493710621053608633701515e-02 5.85471759754055626548407964548915e-04
            1.39846172517608963864477189531499e-02 8.92552710498867973853975854581222e-01 5.85471759754055626548407964548915e-04
            8.92552710498867973853975854581222e-01 9.34626722493710621053608633701515e-02 5.85471759754055626548407964548915e-04
            8.92552710498867973853975854581222e-01 1.39846172517608963864477189531499e-02 5.85471759754055626548407964548915e-04
            3.87408987689019879141483215789776e-01 1.19357764315077496952710589539493e-01 2.70374842821991760027922069298256e-03
            3.87408987689019879141483215789776e-01 4.93233247995902623905806194670731e-01 2.70374842821991760027922069298256e-03
            1.19357764315077496952710589539493e-01 3.87408987689019879141483215789776e-01 2.70374842821991760027922069298256e-03
            1.19357764315077496952710589539493e-01 4.93233247995902623905806194670731e-01 2.70374842821991760027922069298256e-03
            4.93233247995902623905806194670731e-01 3.87408987689019879141483215789776e-01 2.70374842821991760027922069298256e-03
            4.93233247995902623905806194670731e-01 1.19357764315077496952710589539493e-01 2.70374842821991760027922069298256e-03
            4.07184695661520157372592620959040e-01 7.35846549853617637459990419301903e-02 2.26669456187450968007168761175762e-03
            4.07184695661520157372592620959040e-01 5.19230649353118023370257105852943e-01 2.26669456187450968007168761175762e-03
            7.35846549853617637459990419301903e-02 4.07184695661520157372592620959040e-01 2.26669456187450968007168761175762e-03
            7.35846549853617637459990419301903e-02 5.19230649353118023370257105852943e-01 2.26669456187450968007168761175762e-03
            5.19230649353118023370257105852943e-01 4.07184695661520157372592620959040e-01 2.26669456187450968007168761175762e-03
            5.19230649353118023370257105852943e-01 7.35846549853617637459990419301903e-02 2.26669456187450968007168761175762e-03
            1.87640583389709708628245721229177e-01 1.27167478485407797750283975801722e-02 7.63422222268318587980184819485885e-04
            1.87640583389709708628245721229177e-01 7.99642668761749497718938073376194e-01 7.63422222268318587980184819485885e-04
            1.27167478485407797750283975801722e-02 1.87640583389709708628245721229177e-01 7.63422222268318587980184819485885e-04
            1.27167478485407797750283975801722e-02 7.99642668761749497718938073376194e-01 7.63422222268318587980184819485885e-04
            7.99642668761749497718938073376194e-01 1.87640583389709708628245721229177e-01 7.63422222268318587980184819485885e-04
            7.99642668761749497718938073376194e-01 1.27167478485407797750283975801722e-02 7.63422222268318587980184819485885e-04
            1.52225526038276298690377075217839e-01 2.44038040768785029904774930287203e-03 2.99277879376055508907350510838796e-04
            1.52225526038276298690377075217839e-01 8.45334093554035770345933542557759e-01 2.99277879376055508907350510838796e-04
            2.44038040768785029904774930287203e-03 1.52225526038276298690377075217839e-01 2.99277879376055508907350510838796e-04
            2.44038040768785029904774930287203e-03 8.45334093554035770345933542557759e-01 2.99277879376055508907350510838796e-04
            8.45334093554035770345933542557759e-01 1.52225526038276298690377075217839e-01 2.99277879376055508907350510838796e-04
            8.45334093554035770345933542557759e-01 2.44038040768785029904774930287203e-03 2.99277879376055508907350510838796e-04
            1.74142001786563360710147208010312e-01 1.29890625977294477122470084395900e-01 2.11066871570363048110641202015358e-03
            1.74142001786563360710147208010312e-01 6.95967372236142134411807091964874e-01 2.11066871570363048110641202015358e-03
            1.29890625977294477122470084395900e-01 1.74142001786563360710147208010312e-01 2.11066871570363048110641202015358e-03
            1.29890625977294477122470084395900e-01 6.95967372236142134411807091964874e-01 2.11066871570363048110641202015358e-03
            6.95967372236142134411807091964874e-01 1.74142001786563360710147208010312e-01 2.11066871570363048110641202015358e-03
            6.95967372236142134411807091964874e-01 1.29890625977294477122470084395900e-01 2.11066871570363048110641202015358e-03
            3.21408155175526122349083379958756e-01 1.43880630387304531137004914853605e-01 2.75728111566974709925603725935161e-03
            3.21408155175526122349083379958756e-01 5.34711214437169291002760473929811e-01 2.75728111566974709925603725935161e-03
            1.43880630387304531137004914853605e-01 3.21408155175526122349083379958756e-01 2.75728111566974709925603725935161e-03
            1.43880630387304531137004914853605e-01 5.34711214437169291002760473929811e-01 2.75728111566974709925603725935161e-03
            5.34711214437169291002760473929811e-01 3.21408155175526122349083379958756e-01 2.75728111566974709925603725935161e-03
            5.34711214437169291002760473929811e-01 1.43880630387304531137004914853605e-01 2.75728111566974709925603725935161e-03
            8.88043361734264791174098263581982e-02 5.88598092883567541866440819831041e-02 1.16858890495533281062012775919356e-03
            8.88043361734264791174098263581982e-02 8.52335854538216697307007052586414e-01 1.16858890495533281062012775919356e-03
            5.88598092883567541866440819831041e-02 8.88043361734264791174098263581982e-02 1.16858890495533281062012775919356e-03
            5.88598092883567541866440819831041e-02 8.52335854538216697307007052586414e-01 1.16858890495533281062012775919356e-03
            8.52335854538216697307007052586414e-01 8.88043361734264791174098263581982e-02 1.16858890495533281062012775919356e-03
            8.52335854538216697307007052586414e-01 5.88598092883567541866440819831041e-02 1.16858890495533281062012775919356e-03
            1.30428915551712210074697395612020e-01 6.02973413848093622857682305493654e-02 1.39399447633984903752168982293824e-03
            1.30428915551712210074697395612020e-01 8.09273743063478434578428277745843e-01 1.39399447633984903752168982293824e-03
            6.02973413848093622857682305493654e-02 1.30428915551712210074697395612020e-01 1.39399447633984903752168982293824e-03
            6.02973413848093622857682305493654e-02 8.09273743063478434578428277745843e-01 1.39399447633984903752168982293824e-03
            8.09273743063478434578428277745843e-01 1.30428915551712210074697395612020e-01 1.39399447633984903752168982293824e-03
            8.09273743063478434578428277745843e-01 6.02973413848093622857682305493654e-02 1.39399447633984903752168982293824e-03
            1.36837912722210020000090935354820e-01 1.32254525993588060300831799054322e-02 7.15287497685284149283313492873049e-04
            1.36837912722210020000090935354820e-01 8.49936634678431168765655456809327e-01 7.15287497685284149283313492873049e-04
            1.32254525993588060300831799054322e-02 1.36837912722210020000090935354820e-01 7.15287497685284149283313492873049e-04
            1.32254525993588060300831799054322e-02 8.49936634678431168765655456809327e-01 7.15287497685284149283313492873049e-04
            8.49936634678431168765655456809327e-01 1.36837912722210020000090935354820e-01 7.15287497685284149283313492873049e-04
            8.49936634678431168765655456809327e-01 1.32254525993588060300831799054322e-02 7.15287497685284149283313492873049e-04
            3.52853359179670589274024905535043e-01 5.72712595019886847036438837221795e-02 1.98602112093872228484880970711401e-03
            3.52853359179670589274024905535043e-01 5.89875381318340830105739769351203e-01 1.98602112093872228484880970711401e-03
            5.72712595019886847036438837221795e-02 3.52853359179670589274024905535043e-01 1.98602112093872228484880970711401e-03
            5.72712595019886847036438837221795e-02 5.89875381318340830105739769351203e-01 1.98602112093872228484880970711401e-03
            5.89875381318340830105739769351203e-01 3.52853359179670589274024905535043e-01 1.98602112093872228484880970711401e-03
            5.89875381318340830105739769351203e-01 5.72712595019886847036438837221795e-02 1.98602112093872228484880970711401e-03
            2.46386900217142157387684164859820e-01 1.09336475862860226654360573661506e-02 8.11510714695543727836835046218766e-04
            2.46386900217142157387684164859820e-01 7.42679452196571832089944109611679e-01 8.11510714695543727836835046218766e-04
            1.09336475862860226654360573661506e-02 2.46386900217142157387684164859820e-01 8.11510714695543727836835046218766e-04
            1.09336475862860226654360573661506e-02 7.42679452196571832089944109611679e-01 8.11510714695543727836835046218766e-04
            7.42679452196571832089944109611679e-01 2.46386900217142157387684164859820e-01 8.11510714695543727836835046218766e-04
            7.42679452196571832089944109611679e-01 1.09336475862860226654360573661506e-02 8.11510714695543727836835046218766e-04
            3.43520044059471285624596248453599e-01 9.59251650895475566072079232071701e-02 2.41542743894754448341632979690985e-03
            3.43520044059471285624596248453599e-01 5.60554790850981254912710483040428e-01 2.41542743894754448341632979690985e-03
            9.59251650895475566072079232071701e-02 3.43520044059471285624596248453599e-01 2.41542743894754448341632979690985e-03
            9.59251650895475566072079232071701e-02 5.60554790850981254912710483040428e-01 2.41542743894754448341632979690985e-03
            5.60554790850981254912710483040428e-01 3.43520044059471285624596248453599e-01 2.41542743894754448341632979690985e-03
            5.60554790850981254912710483040428e-01 9.59251650895475566072079232071701e-02 2.41542743894754448341632979690985e-03
            4.16208963636464202018316882458748e-01 3.96704297157567664644517435590387e-02 1.76983940807325101918223708707956e-03
            4.16208963636464202018316882458748e-01 5.44120606647779059272806989611126e-01 1.76983940807325101918223708707956e-03
            3.96704297157567664644517435590387e-02 4.16208963636464202018316882458748e-01 1.76983940807325101918223708707956e-03
            3.96704297157567664644517435590387e-02 5.44120606647779059272806989611126e-01 1.76983940807325101918223708707956e-03
            5.44120606647779059272806989611126e-01 4.16208963636464202018316882458748e-01 1.76983940807325101918223708707956e-03
            5.44120606647779059272806989611126e-01 3.96704297157567664644517435590387e-02 1.76983940807325101918223708707956e-03
            3.90118106999857716665758289309451e-02 2.79504775052941568241893222079852e-03 1.83532994709030615552297627246503e-04
            3.90118106999857716665758289309451e-02 9.58193141549484872498965160048101e-01 1.83532994709030615552297627246503e-04
            2.79504775052941568241893222079852e-03 3.90118106999857716665758289309451e-02 1.83532994709030615552297627246503e-04
            2.79504775052941568241893222079852e-03 9.58193141549484872498965160048101e-01 1.83532994709030615552297627246503e-04
            9.58193141549484872498965160048101e-01 3.90118106999857716665758289309451e-02 1.83532994709030615552297627246503e-04
            9.58193141549484872498965160048101e-01 2.79504775052941568241893222079852e-03 1.83532994709030615552297627246503e-04
            1.07180714134606072662769804537675e-01 2.66661165267688318042393902373988e-03 2.85484954702090840301942353818276e-04
            1.07180714134606072662769804537675e-01 8.90152674212717043289444518450182e-01 2.85484954702090840301942353818276e-04
            2.66661165267688318042393902373988e-03 1.07180714134606072662769804537675e-01 2.85484954702090840301942353818276e-04
            2.66661165267688318042393902373988e-03 8.90152674212717043289444518450182e-01 2.85484954702090840301942353818276e-04
            8.90152674212717043289444518450182e-01 1.07180714134606072662769804537675e-01 2.85484954702090840301942353818276e-04
            8.90152674212717043289444518450182e-01 2.66661165267688318042393902373988e-03 2.85484954702090840301942353818276e-04
            4.26423819706326090450687615884817e-01 2.53953007625778645617309337012557e-03 4.38056769584410936237833800177555e-04
            4.26423819706326090450687615884817e-01 5.71036650217416119623692338791443e-01 4.38056769584410936237833800177555e-04
            2.53953007625778645617309337012557e-03 4.26423819706326090450687615884817e-01 4.38056769584410936237833800177555e-04
            2.53953007625778645617309337012557e-03 5.71036650217416119623692338791443e-01 4.38056769584410936237833800177555e-04
            5.71036650217416119623692338791443e-01 4.26423819706326090450687615884817e-01 4.38056769584410936237833800177555e-04
            5.71036650217416119623692338791443e-01 2.53953007625778645617309337012557e-03 4.38056769584410936237833800177555e-04
            3.51960131363829642481277915067039e-01 2.87313755582887971062877596750695e-02 1.54355548746208225824316784269286e-03
            3.51960131363829642481277915067039e-01 6.19308493077881605515244700654875e-01 1.54355548746208225824316784269286e-03
            2.87313755582887971062877596750695e-02 3.51960131363829642481277915067039e-01 1.54355548746208225824316784269286e-03
            2.87313755582887971062877596750695e-02 6.19308493077881605515244700654875e-01 1.54355548746208225824316784269286e-03
            6.19308493077881605515244700654875e-01 3.51960131363829642481277915067039e-01 1.54355548746208225824316784269286e-03
            6.19308493077881605515244700654875e-01 2.87313755582887971062877596750695e-02 1.54355548746208225824316784269286e-03
            ];
        
    case  46
        
        % ALG. DEG.:   46
        % PTS CARD.:  385
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 2.4e-15
        
        xyw=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.31752356160359409761873017430389e-03
            4.79162709481021131807665369706228e-01 4.79162709481021131807665369706228e-01 7.53181193949794603771652479196064e-04
            4.79162709481021131807665369706228e-01 4.16745810379577363846692605875432e-02 7.53181193949794603771652479196064e-04
            4.16745810379577363846692605875432e-02 4.79162709481021131807665369706228e-01 7.53181193949794603771652479196064e-04
            4.86830795833199034272631422481936e-02 4.86830795833199034272631422481936e-02 6.23734590822673908364071060361766e-04
            4.86830795833199034272631422481936e-02 9.02633840833360179267685907689156e-01 6.23734590822673908364071060361766e-04
            9.02633840833360179267685907689156e-01 4.86830795833199034272631422481936e-02 6.23734590822673908364071060361766e-04
            4.86849434452578500032160491173272e-01 4.86849434452578500032160491173272e-01 1.09475819165005838241866431559401e-03
            4.86849434452578500032160491173272e-01 2.63011310948429999356790176534560e-02 1.09475819165005838241866431559401e-03
            2.63011310948429999356790176534560e-02 4.86849434452578500032160491173272e-01 1.09475819165005838241866431559401e-03
            1.96294809229449501719599879834277e-01 1.96294809229449501719599879834277e-01 2.26630825917092460289326005806743e-03
            1.96294809229449501719599879834277e-01 6.07410381541100941049649009073619e-01 2.26630825917092460289326005806743e-03
            6.07410381541100941049649009073619e-01 1.96294809229449501719599879834277e-01 2.26630825917092460289326005806743e-03
            1.65798716567095938412279565454810e-01 1.65798716567095938412279565454810e-01 2.05606677560805613594330232274388e-03
            1.65798716567095938412279565454810e-01 6.68402566865808123175440869090380e-01 2.05606677560805613594330232274388e-03
            6.68402566865808123175440869090380e-01 1.65798716567095938412279565454810e-01 2.05606677560805613594330232274388e-03
            2.29007117787044606194157836398517e-01 2.29007117787044606194157836398517e-01 2.66814199893167149774120261440657e-03
            2.29007117787044606194157836398517e-01 5.41985764425910732100533095945138e-01 2.66814199893167149774120261440657e-03
            5.41985764425910732100533095945138e-01 2.29007117787044606194157836398517e-01 2.66814199893167149774120261440657e-03
            2.99898101879728595786644973486545e-01 2.99898101879728595786644973486545e-01 3.23781096391132254194711315165023e-03
            2.99898101879728595786644973486545e-01 4.00203796240542808426710053026909e-01 3.23781096391132254194711315165023e-03
            4.00203796240542808426710053026909e-01 2.99898101879728595786644973486545e-01 3.23781096391132254194711315165023e-03
            3.66081261411425384899587243126007e-01 3.66081261411425384899587243126007e-01 3.26415847271572211185652889753328e-03
            3.66081261411425384899587243126007e-01 2.67837477177149230200825513747986e-01 3.26415847271572211185652889753328e-03
            2.67837477177149230200825513747986e-01 3.66081261411425384899587243126007e-01 3.26415847271572211185652889753328e-03
            3.07693439520817652033102262976172e-02 3.07693439520817652033102262976172e-02 4.85854850068563419700912175969165e-04
            3.07693439520817652033102262976172e-02 9.38461312095836497348955163033679e-01 4.85854850068563419700912175969165e-04
            9.38461312095836497348955163033679e-01 3.07693439520817652033102262976172e-02 4.85854850068563419700912175969165e-04
            8.24986937635582839867254278942710e-02 8.24986937635582839867254278942710e-02 1.21934895073771390645867018776016e-03
            8.24986937635582839867254278942710e-02 8.35002612472883432026549144211458e-01 1.21934895073771390645867018776016e-03
            8.35002612472883432026549144211458e-01 8.24986937635582839867254278942710e-02 1.21934895073771390645867018776016e-03
            4.60823004347588105478195075193071e-01 4.60823004347588105478195075193071e-01 2.09410203588820389639102437229212e-03
            4.60823004347588105478195075193071e-01 7.83539913048237890436098496138584e-02 2.09410203588820389639102437229212e-03
            7.83539913048237890436098496138584e-02 4.60823004347588105478195075193071e-01 2.09410203588820389639102437229212e-03
            2.66615708697696740525628911200329e-01 2.66615708697696740525628911200329e-01 2.99313714823756229707618636837196e-03
            2.66615708697696740525628911200329e-01 4.66768582604606518948742177599343e-01 2.99313714823756229707618636837196e-03
            4.66768582604606518948742177599343e-01 2.66615708697696740525628911200329e-01 2.99313714823756229707618636837196e-03
            1.28439434947512254314716528824647e-01 1.28439434947512254314716528824647e-01 1.88674007826654041575231701699522e-03
            1.28439434947512254314716528824647e-01 7.43121130104975491370566942350706e-01 1.88674007826654041575231701699522e-03
            7.43121130104975491370566942350706e-01 1.28439434947512254314716528824647e-01 1.88674007826654041575231701699522e-03
            3.97511575402048289440415373974247e-01 3.97511575402048289440415373974247e-01 3.24082362724328280206842478605722e-03
            3.97511575402048289440415373974247e-01 2.04976849195903421119169252051506e-01 3.24082362724328280206842478605722e-03
            2.04976849195903421119169252051506e-01 3.97511575402048289440415373974247e-01 3.24082362724328280206842478605722e-03
            1.28796984358850331170431857685799e-02 1.28796984358850331170431857685799e-02 2.24673355140023097440207799913026e-04
            1.28796984358850331170431857685799e-02 9.74240603128229909479784964787541e-01 2.24673355140023097440207799913026e-04
            9.74240603128229909479784964787541e-01 1.28796984358850331170431857685799e-02 2.24673355140023097440207799913026e-04
            4.92922242405799231057539344647012e-04 4.92922242405799231057539344647012e-04 7.04761006797527232584435474826989e-06
            4.92922242405799231057539344647012e-04 9.99014155515188351230904117983300e-01 7.04761006797527232584435474826989e-06
            9.99014155515188351230904117983300e-01 4.92922242405799231057539344647012e-04 7.04761006797527232584435474826989e-06
            3.97969764640709822156594555053744e-01 1.54123149166077250127315778627235e-01 2.22685409346650830392233544330338e-03
            3.97969764640709822156594555053744e-01 4.47907086193212844449362819432281e-01 2.22685409346650830392233544330338e-03
            1.54123149166077250127315778627235e-01 3.97969764640709822156594555053744e-01 2.22685409346650830392233544330338e-03
            1.54123149166077250127315778627235e-01 4.47907086193212844449362819432281e-01 2.22685409346650830392233544330338e-03
            4.47907086193212844449362819432281e-01 3.97969764640709822156594555053744e-01 2.22685409346650830392233544330338e-03
            4.47907086193212844449362819432281e-01 1.54123149166077250127315778627235e-01 2.22685409346650830392233544330338e-03
            4.14307607988175785607154466561042e-01 1.13136705924023417413337710968335e-01 2.16000476232082504537967970748014e-03
            4.14307607988175785607154466561042e-01 4.72555686087800796979507822470623e-01 2.16000476232082504537967970748014e-03
            1.13136705924023417413337710968335e-01 4.14307607988175785607154466561042e-01 2.16000476232082504537967970748014e-03
            1.13136705924023417413337710968335e-01 4.72555686087800796979507822470623e-01 2.16000476232082504537967970748014e-03
            4.72555686087800796979507822470623e-01 4.14307607988175785607154466561042e-01 2.16000476232082504537967970748014e-03
            4.72555686087800796979507822470623e-01 1.13136705924023417413337710968335e-01 2.16000476232082504537967970748014e-03
            3.29728249876623691339005972622545e-01 7.30183277797032820322087331987859e-02 1.54343982796946855955144695116132e-03
            3.29728249876623691339005972622545e-01 5.97253422343673068262148717622040e-01 1.54343982796946855955144695116132e-03
            7.30183277797032820322087331987859e-02 3.29728249876623691339005972622545e-01 1.54343982796946855955144695116132e-03
            7.30183277797032820322087331987859e-02 5.97253422343673068262148717622040e-01 1.54343982796946855955144695116132e-03
            5.97253422343673068262148717622040e-01 3.29728249876623691339005972622545e-01 1.54343982796946855955144695116132e-03
            5.97253422343673068262148717622040e-01 7.30183277797032820322087331987859e-02 1.54343982796946855955144695116132e-03
            2.25759242330057757763839276776707e-01 1.09803376807363559097829863731022e-01 1.71511290774017076399271530817714e-03
            2.25759242330057757763839276776707e-01 6.64437380862578752527269898564555e-01 1.71511290774017076399271530817714e-03
            1.09803376807363559097829863731022e-01 2.25759242330057757763839276776707e-01 1.71511290774017076399271530817714e-03
            1.09803376807363559097829863731022e-01 6.64437380862578752527269898564555e-01 1.71511290774017076399271530817714e-03
            6.64437380862578752527269898564555e-01 2.25759242330057757763839276776707e-01 1.71511290774017076399271530817714e-03
            6.64437380862578752527269898564555e-01 1.09803376807363559097829863731022e-01 1.71511290774017076399271530817714e-03
            3.28699347560523658096087729063584e-01 1.02835875371072957285001336913410e-01 1.89138099119563730075854746104369e-03
            3.28699347560523658096087729063584e-01 5.68464777068403370741123126208549e-01 1.89138099119563730075854746104369e-03
            1.02835875371072957285001336913410e-01 3.28699347560523658096087729063584e-01 1.89138099119563730075854746104369e-03
            1.02835875371072957285001336913410e-01 5.68464777068403370741123126208549e-01 1.89138099119563730075854746104369e-03
            5.68464777068403370741123126208549e-01 3.28699347560523658096087729063584e-01 1.89138099119563730075854746104369e-03
            5.68464777068403370741123126208549e-01 1.02835875371072957285001336913410e-01 1.89138099119563730075854746104369e-03
            5.70245183292808974218068840400520e-02 1.04550153444100853150633412269599e-02 3.35215320031305257213877224486964e-04
            5.70245183292808974218068840400520e-02 9.32520466326309027671470630593831e-01 3.35215320031305257213877224486964e-04
            1.04550153444100853150633412269599e-02 5.70245183292808974218068840400520e-02 3.35215320031305257213877224486964e-04
            1.04550153444100853150633412269599e-02 9.32520466326309027671470630593831e-01 3.35215320031305257213877224486964e-04
            9.32520466326309027671470630593831e-01 5.70245183292808974218068840400520e-02 3.35215320031305257213877224486964e-04
            9.32520466326309027671470630593831e-01 1.04550153444100853150633412269599e-02 3.35215320031305257213877224486964e-04
            2.18153490627607082874561683638603e-01 1.47522261546056410175253859051736e-01 2.08705738367027503893380568911198e-03
            2.18153490627607082874561683638603e-01 6.34324247826336562461335688567488e-01 2.08705738367027503893380568911198e-03
            1.47522261546056410175253859051736e-01 2.18153490627607082874561683638603e-01 2.08705738367027503893380568911198e-03
            1.47522261546056410175253859051736e-01 6.34324247826336562461335688567488e-01 2.08705738367027503893380568911198e-03
            6.34324247826336562461335688567488e-01 2.18153490627607082874561683638603e-01 2.08705738367027503893380568911198e-03
            6.34324247826336562461335688567488e-01 1.47522261546056410175253859051736e-01 2.08705738367027503893380568911198e-03
            2.02306345883463578605443444757839e-01 5.21260968074300848340563163674233e-02 1.27918332848801557730111788657723e-03
            2.02306345883463578605443444757839e-01 7.45567557309106399010545374039793e-01 1.27918332848801557730111788657723e-03
            5.21260968074300848340563163674233e-02 2.02306345883463578605443444757839e-01 1.27918332848801557730111788657723e-03
            5.21260968074300848340563163674233e-02 7.45567557309106399010545374039793e-01 1.27918332848801557730111788657723e-03
            7.45567557309106399010545374039793e-01 2.02306345883463578605443444757839e-01 1.27918332848801557730111788657723e-03
            7.45567557309106399010545374039793e-01 5.21260968074300848340563163674233e-02 1.27918332848801557730111788657723e-03
            2.13283209808807994178891931369435e-01 8.00841957661688891301210446727055e-02 1.59199347606164918339510805367354e-03
            2.13283209808807994178891931369435e-01 7.06632594425023130568774831772316e-01 1.59199347606164918339510805367354e-03
            8.00841957661688891301210446727055e-02 2.13283209808807994178891931369435e-01 1.59199347606164918339510805367354e-03
            8.00841957661688891301210446727055e-02 7.06632594425023130568774831772316e-01 1.59199347606164918339510805367354e-03
            7.06632594425023130568774831772316e-01 2.13283209808807994178891931369435e-01 1.59199347606164918339510805367354e-03
            7.06632594425023130568774831772316e-01 8.00841957661688891301210446727055e-02 1.59199347606164918339510805367354e-03
            8.21335887122196829646103566346937e-03 2.16078035466830617461475583240826e-03 6.58585463976957708603920949030908e-05
            8.21335887122196829646103566346937e-03 9.89625860774109744610882444249000e-01 6.58585463976957708603920949030908e-05
            2.16078035466830617461475583240826e-03 8.21335887122196829646103566346937e-03 6.58585463976957708603920949030908e-05
            2.16078035466830617461475583240826e-03 9.89625860774109744610882444249000e-01 6.58585463976957708603920949030908e-05
            9.89625860774109744610882444249000e-01 8.21335887122196829646103566346937e-03 6.58585463976957708603920949030908e-05
            9.89625860774109744610882444249000e-01 2.16078035466830617461475583240826e-03 6.58585463976957708603920949030908e-05
            4.61515361612300334392955392104341e-01 1.13638023497272810935498910112074e-02 8.22836191553004978396779200977562e-04
            4.61515361612300334392955392104341e-01 5.27120836037972440024645948142279e-01 8.22836191553004978396779200977562e-04
            1.13638023497272810935498910112074e-02 4.61515361612300334392955392104341e-01 8.22836191553004978396779200977562e-04
            1.13638023497272810935498910112074e-02 5.27120836037972440024645948142279e-01 8.22836191553004978396779200977562e-04
            5.27120836037972440024645948142279e-01 4.61515361612300334392955392104341e-01 8.22836191553004978396779200977562e-04
            5.27120836037972440024645948142279e-01 1.13638023497272810935498910112074e-02 8.22836191553004978396779200977562e-04
            3.54622467312705691444563171899063e-01 1.29778523617651048160936966269219e-01 2.29774154662928681985833989642742e-03
            3.54622467312705691444563171899063e-01 5.15599009069643288150075477460632e-01 2.29774154662928681985833989642742e-03
            1.29778523617651048160936966269219e-01 3.54622467312705691444563171899063e-01 2.29774154662928681985833989642742e-03
            1.29778523617651048160936966269219e-01 5.15599009069643288150075477460632e-01 2.29774154662928681985833989642742e-03
            5.15599009069643288150075477460632e-01 3.54622467312705691444563171899063e-01 2.29774154662928681985833989642742e-03
            5.15599009069643288150075477460632e-01 1.29778523617651048160936966269219e-01 2.29774154662928681985833989642742e-03
            3.10920643586031608673891923899646e-01 4.89372786072454005190657255752740e-02 1.43443007840781669090524097498474e-03
            3.10920643586031608673891923899646e-01 6.40142077806722942234785023174481e-01 1.43443007840781669090524097498474e-03
            4.89372786072454005190657255752740e-02 3.10920643586031608673891923899646e-01 1.43443007840781669090524097498474e-03
            4.89372786072454005190657255752740e-02 6.40142077806722942234785023174481e-01 1.43443007840781669090524097498474e-03
            6.40142077806722942234785023174481e-01 3.10920643586031608673891923899646e-01 1.43443007840781669090524097498474e-03
            6.40142077806722942234785023174481e-01 4.89372786072454005190657255752740e-02 1.43443007840781669090524097498474e-03
            3.04955784133230656252067092282232e-01 1.58417559007465957865434802442905e-01 2.36336703046012347115278906528602e-03
            3.04955784133230656252067092282232e-01 5.36626656859303441393649336532690e-01 2.36336703046012347115278906528602e-03
            1.58417559007465957865434802442905e-01 3.04955784133230656252067092282232e-01 2.36336703046012347115278906528602e-03
            1.58417559007465957865434802442905e-01 5.36626656859303441393649336532690e-01 2.36336703046012347115278906528602e-03
            5.36626656859303441393649336532690e-01 3.04955784133230656252067092282232e-01 2.36336703046012347115278906528602e-03
            5.36626656859303441393649336532690e-01 1.58417559007465957865434802442905e-01 2.36336703046012347115278906528602e-03
            2.50794437563904126786695769624203e-01 1.75500085438409114679814138071379e-01 2.40436588441476743749358746526923e-03
            2.50794437563904126786695769624203e-01 5.73705476997686703022338861046592e-01 2.40436588441476743749358746526923e-03
            1.75500085438409114679814138071379e-01 2.50794437563904126786695769624203e-01 2.40436588441476743749358746526923e-03
            1.75500085438409114679814138071379e-01 5.73705476997686703022338861046592e-01 2.40436588441476743749358746526923e-03
            5.73705476997686703022338861046592e-01 2.50794437563904126786695769624203e-01 2.40436588441476743749358746526923e-03
            5.73705476997686703022338861046592e-01 1.75500085438409114679814138071379e-01 2.40436588441476743749358746526923e-03
            4.33869221864480514305739689007169e-01 5.31263813104962887012838734790421e-02 1.59638354184673180932718672409010e-03
            4.33869221864480514305739689007169e-01 5.13004396825023190054082533606561e-01 1.59638354184673180932718672409010e-03
            5.31263813104962887012838734790421e-02 4.33869221864480514305739689007169e-01 1.59638354184673180932718672409010e-03
            5.31263813104962887012838734790421e-02 5.13004396825023190054082533606561e-01 1.59638354184673180932718672409010e-03
            5.13004396825023190054082533606561e-01 4.33869221864480514305739689007169e-01 1.59638354184673180932718672409010e-03
            5.13004396825023190054082533606561e-01 5.31263813104962887012838734790421e-02 1.59638354184673180932718672409010e-03
            7.70376986253196172071966429939494e-02 5.09089773583295684966643079860660e-02 8.74382525144719422068029235362019e-04
            7.70376986253196172071966429939494e-02 8.72053324016350828173926856834441e-01 8.74382525144719422068029235362019e-04
            5.09089773583295684966643079860660e-02 7.70376986253196172071966429939494e-02 8.74382525144719422068029235362019e-04
            5.09089773583295684966643079860660e-02 8.72053324016350828173926856834441e-01 8.74382525144719422068029235362019e-04
            8.72053324016350828173926856834441e-01 7.70376986253196172071966429939494e-02 8.74382525144719422068029235362019e-04
            8.72053324016350828173926856834441e-01 5.09089773583295684966643079860660e-02 8.74382525144719422068029235362019e-04
            2.76836186019522423329419780202443e-01 1.11213995971377495319831751885431e-02 7.15335166191507490149326109474259e-04
            2.76836186019522423329419780202443e-01 7.12042414383339816730256188748172e-01 7.15335166191507490149326109474259e-04
            1.11213995971377495319831751885431e-02 2.76836186019522423329419780202443e-01 7.15335166191507490149326109474259e-04
            1.11213995971377495319831751885431e-02 7.12042414383339816730256188748172e-01 7.15335166191507490149326109474259e-04
            7.12042414383339816730256188748172e-01 2.76836186019522423329419780202443e-01 7.15335166191507490149326109474259e-04
            7.12042414383339816730256188748172e-01 1.11213995971377495319831751885431e-02 7.15335166191507490149326109474259e-04
            1.73637162103421421877058605787170e-01 1.17797536807179720197424899197358e-02 6.34087639467668354349705328587561e-04
            1.73637162103421421877058605787170e-01 8.14583084215860675492137943365378e-01 6.34087639467668354349705328587561e-04
            1.17797536807179720197424899197358e-02 1.73637162103421421877058605787170e-01 6.34087639467668354349705328587561e-04
            1.17797536807179720197424899197358e-02 8.14583084215860675492137943365378e-01 6.34087639467668354349705328587561e-04
            8.14583084215860675492137943365378e-01 1.73637162103421421877058605787170e-01 6.34087639467668354349705328587561e-04
            8.14583084215860675492137943365378e-01 1.17797536807179720197424899197358e-02 6.34087639467668354349705328587561e-04
            3.35433688661376960826743243160308e-01 1.18081263036933694493546553871965e-02 7.89516871955841929332786666151378e-04
            3.35433688661376960826743243160308e-01 6.52758185034929772072587184084114e-01 7.89516871955841929332786666151378e-04
            1.18081263036933694493546553871965e-02 3.35433688661376960826743243160308e-01 7.89516871955841929332786666151378e-04
            1.18081263036933694493546553871965e-02 6.52758185034929772072587184084114e-01 7.89516871955841929332786666151378e-04
            6.52758185034929772072587184084114e-01 3.35433688661376960826743243160308e-01 7.89516871955841929332786666151378e-04
            6.52758185034929772072587184084114e-01 1.18081263036933694493546553871965e-02 7.89516871955841929332786666151378e-04
            3.75064218452396624936540092676296e-01 5.12438839908936349942791821376886e-02 1.60772260421603485798514032723006e-03
            3.75064218452396624936540092676296e-01 5.73691897556709684558029493928188e-01 1.60772260421603485798514032723006e-03
            5.12438839908936349942791821376886e-02 3.75064218452396624936540092676296e-01 1.60772260421603485798514032723006e-03
            5.12438839908936349942791821376886e-02 5.73691897556709684558029493928188e-01 1.60772260421603485798514032723006e-03
            5.73691897556709684558029493928188e-01 3.75064218452396624936540092676296e-01 1.60772260421603485798514032723006e-03
            5.73691897556709684558029493928188e-01 5.12438839908936349942791821376886e-02 1.60772260421603485798514032723006e-03
            3.46711286129900431163264329370577e-01 1.82541256476132840447235139436089e-01 2.69885291100849114900861458465897e-03
            3.46711286129900431163264329370577e-01 4.70747457393966728389500531193335e-01 2.69885291100849114900861458465897e-03
            1.82541256476132840447235139436089e-01 3.46711286129900431163264329370577e-01 2.69885291100849114900861458465897e-03
            1.82541256476132840447235139436089e-01 4.70747457393966728389500531193335e-01 2.69885291100849114900861458465897e-03
            4.70747457393966728389500531193335e-01 3.46711286129900431163264329370577e-01 2.69885291100849114900861458465897e-03
            4.70747457393966728389500531193335e-01 1.82541256476132840447235139436089e-01 2.69885291100849114900861458465897e-03
            2.55968511260664111084395244688494e-01 5.39018714502928694698447031896649e-02 1.53135260150544716059584082756828e-03
            2.55968511260664111084395244688494e-01 6.90129617289042984751290532585699e-01 1.53135260150544716059584082756828e-03
            5.39018714502928694698447031896649e-02 2.55968511260664111084395244688494e-01 1.53135260150544716059584082756828e-03
            5.39018714502928694698447031896649e-02 6.90129617289042984751290532585699e-01 1.53135260150544716059584082756828e-03
            6.90129617289042984751290532585699e-01 2.55968511260664111084395244688494e-01 1.53135260150544716059584082756828e-03
            6.90129617289042984751290532585699e-01 5.39018714502928694698447031896649e-02 1.53135260150544716059584082756828e-03
            2.92006920788531632027229534287471e-01 2.73433134615536990730255695325468e-02 1.15960920481873079948553773022013e-03
            2.92006920788531632027229534287471e-01 6.80649765749914648083063184458297e-01 1.15960920481873079948553773022013e-03
            2.73433134615536990730255695325468e-02 2.92006920788531632027229534287471e-01 1.15960920481873079948553773022013e-03
            2.73433134615536990730255695325468e-02 6.80649765749914648083063184458297e-01 1.15960920481873079948553773022013e-03
            6.80649765749914648083063184458297e-01 2.92006920788531632027229534287471e-01 1.15960920481873079948553773022013e-03
            6.80649765749914648083063184458297e-01 2.73433134615536990730255695325468e-02 1.15960920481873079948553773022013e-03
            2.36417239579575555463719638282782e-01 2.97423818792802684141118874094900e-02 1.14805014508864732837578603863449e-03
            2.36417239579575555463719638282782e-01 7.33840378541144144897145906725200e-01 1.14805014508864732837578603863449e-03
            2.97423818792802684141118874094900e-02 2.36417239579575555463719638282782e-01 1.14805014508864732837578603863449e-03
            2.97423818792802684141118874094900e-02 7.33840378541144144897145906725200e-01 1.14805014508864732837578603863449e-03
            7.33840378541144144897145906725200e-01 2.36417239579575555463719638282782e-01 1.14805014508864732837578603863449e-03
            7.33840378541144144897145906725200e-01 2.97423818792802684141118874094900e-02 1.14805014508864732837578603863449e-03
            4.66150898129435442562851221737219e-01 2.19129468105736533237171137500354e-03 3.66905166035601769392027193816830e-04
            4.66150898129435442562851221737219e-01 5.31657807189507169987052748183487e-01 3.66905166035601769392027193816830e-04
            2.19129468105736533237171137500354e-03 4.66150898129435442562851221737219e-01 3.66905166035601769392027193816830e-04
            2.19129468105736533237171137500354e-03 5.31657807189507169987052748183487e-01 3.66905166035601769392027193816830e-04
            5.31657807189507169987052748183487e-01 4.66150898129435442562851221737219e-01 3.66905166035601769392027193816830e-04
            5.31657807189507169987052748183487e-01 2.19129468105736533237171137500354e-03 3.66905166035601769392027193816830e-04
            2.23482629145585642849169971668744e-01 1.25736658366042697299036490221624e-02 7.21880138370406434882020629828503e-04
            2.23482629145585642849169971668744e-01 7.63943705017810170687653226195835e-01 7.21880138370406434882020629828503e-04
            1.25736658366042697299036490221624e-02 2.23482629145585642849169971668744e-01 7.21880138370406434882020629828503e-04
            1.25736658366042697299036490221624e-02 7.63943705017810170687653226195835e-01 7.21880138370406434882020629828503e-04
            7.63943705017810170687653226195835e-01 2.23482629145585642849169971668744e-01 7.21880138370406434882020629828503e-04
            7.63943705017810170687653226195835e-01 1.25736658366042697299036490221624e-02 7.21880138370406434882020629828503e-04
            3.97368013878000581051708195445826e-01 1.15196691295125427934031847598817e-02 8.29333161982134217833639855399497e-04
            3.97368013878000581051708195445826e-01 5.91112316992486896971570331515977e-01 8.29333161982134217833639855399497e-04
            1.15196691295125427934031847598817e-02 3.97368013878000581051708195445826e-01 8.29333161982134217833639855399497e-04
            1.15196691295125427934031847598817e-02 5.91112316992486896971570331515977e-01 8.29333161982134217833639855399497e-04
            5.91112316992486896971570331515977e-01 3.97368013878000581051708195445826e-01 8.29333161982134217833639855399497e-04
            5.91112316992486896971570331515977e-01 1.15196691295125427934031847598817e-02 8.29333161982134217833639855399497e-04
            2.85679243244989944372491663671099e-01 2.10668037666766339022217380261282e-01 2.78536465405663721806028831906588e-03
            2.85679243244989944372491663671099e-01 5.03652719088243716605290956067620e-01 2.78536465405663721806028831906588e-03
            2.10668037666766339022217380261282e-01 2.85679243244989944372491663671099e-01 2.78536465405663721806028831906588e-03
            2.10668037666766339022217380261282e-01 5.03652719088243716605290956067620e-01 2.78536465405663721806028831906588e-03
            5.03652719088243716605290956067620e-01 2.85679243244989944372491663671099e-01 2.78536465405663721806028831906588e-03
            5.03652719088243716605290956067620e-01 2.10668037666766339022217380261282e-01 2.78536465405663721806028831906588e-03
            4.00991912296406793903003062951029e-01 2.17085028246803788537500068400732e-03 3.60718717232300892412688586574632e-04
            4.00991912296406793903003062951029e-01 5.96837237421125266223498329054564e-01 3.60718717232300892412688586574632e-04
            2.17085028246803788537500068400732e-03 4.00991912296406793903003062951029e-01 3.60718717232300892412688586574632e-04
            2.17085028246803788537500068400732e-03 5.96837237421125266223498329054564e-01 3.60718717232300892412688586574632e-04
            5.96837237421125266223498329054564e-01 4.00991912296406793903003062951029e-01 3.60718717232300892412688586574632e-04
            5.96837237421125266223498329054564e-01 2.17085028246803788537500068400732e-03 3.60718717232300892412688586574632e-04
            1.56273510404076748647739236730558e-01 5.57664223797979460850449129338813e-02 1.31106391220739350485235963361674e-03
            1.56273510404076748647739236730558e-01 7.87960067216125326083897562057246e-01 1.31106391220739350485235963361674e-03
            5.57664223797979460850449129338813e-02 1.56273510404076748647739236730558e-01 1.31106391220739350485235963361674e-03
            5.57664223797979460850449129338813e-02 7.87960067216125326083897562057246e-01 1.31106391220739350485235963361674e-03
            7.87960067216125326083897562057246e-01 1.56273510404076748647739236730558e-01 1.31106391220739350485235963361674e-03
            7.87960067216125326083897562057246e-01 5.57664223797979460850449129338813e-02 1.31106391220739350485235963361674e-03
            4.77858936846818488364263544099231e-02 2.09790827030163079536251302670280e-03 1.51789801686191898029767988198557e-04
            4.77858936846818488364263544099231e-02 9.50116198045016457918165997398319e-01 1.51789801686191898029767988198557e-04
            2.09790827030163079536251302670280e-03 4.77858936846818488364263544099231e-02 1.51789801686191898029767988198557e-04
            2.09790827030163079536251302670280e-03 9.50116198045016457918165997398319e-01 1.51789801686191898029767988198557e-04
            9.50116198045016457918165997398319e-01 4.77858936846818488364263544099231e-02 1.51789801686191898029767988198557e-04
            9.50116198045016457918165997398319e-01 2.09790827030163079536251302670280e-03 1.51789801686191898029767988198557e-04
            1.82249159931374082876942566144862e-01 2.90781975795925198835512048844976e-02 1.05004350057420845626632299740777e-03
            1.82249159931374082876942566144862e-01 7.88672642489033459689551364135696e-01 1.05004350057420845626632299740777e-03
            2.90781975795925198835512048844976e-02 1.82249159931374082876942566144862e-01 1.05004350057420845626632299740777e-03
            2.90781975795925198835512048844976e-02 7.88672642489033459689551364135696e-01 1.05004350057420845626632299740777e-03
            7.88672642489033459689551364135696e-01 1.82249159931374082876942566144862e-01 1.05004350057420845626632299740777e-03
            7.88672642489033459689551364135696e-01 2.90781975795925198835512048844976e-02 1.05004350057420845626632299740777e-03
            8.93475681919859154866969674912980e-02 1.15829668440252560313030727456862e-02 4.83149309370667909887403501656422e-04
            8.93475681919859154866969674912980e-02 8.99069464963988762562507872644346e-01 4.83149309370667909887403501656422e-04
            1.15829668440252560313030727456862e-02 8.93475681919859154866969674912980e-02 4.83149309370667909887403501656422e-04
            1.15829668440252560313030727456862e-02 8.99069464963988762562507872644346e-01 4.83149309370667909887403501656422e-04
            8.99069464963988762562507872644346e-01 8.93475681919859154866969674912980e-02 4.83149309370667909887403501656422e-04
            8.99069464963988762562507872644346e-01 1.15829668440252560313030727456862e-02 4.83149309370667909887403501656422e-04
            1.65219881239912769199662534447270e-01 8.74623616502727824206786522154289e-02 1.60233618511449425747539265785235e-03
            1.65219881239912769199662534447270e-01 7.47317757109814406746295389893930e-01 1.60233618511449425747539265785235e-03
            8.74623616502727824206786522154289e-02 1.65219881239912769199662534447270e-01 1.60233618511449425747539265785235e-03
            8.74623616502727824206786522154289e-02 7.47317757109814406746295389893930e-01 1.60233618511449425747539265785235e-03
            7.47317757109814406746295389893930e-01 1.65219881239912769199662534447270e-01 1.60233618511449425747539265785235e-03
            7.47317757109814406746295389893930e-01 8.74623616502727824206786522154289e-02 1.60233618511449425747539265785235e-03
            8.05280047391684350444052142847795e-02 2.07850359821926643222322894644094e-03 1.96147522693671742376880784419768e-04
            8.05280047391684350444052142847795e-02 9.17393491662612303727541984699201e-01 1.96147522693671742376880784419768e-04
            2.07850359821926643222322894644094e-03 8.05280047391684350444052142847795e-02 1.96147522693671742376880784419768e-04
            2.07850359821926643222322894644094e-03 9.17393491662612303727541984699201e-01 1.96147522693671742376880784419768e-04
            9.17393491662612303727541984699201e-01 8.05280047391684350444052142847795e-02 1.96147522693671742376880784419768e-04
            9.17393491662612303727541984699201e-01 2.07850359821926643222322894644094e-03 1.96147522693671742376880784419768e-04
            3.30541501567599038668987532219035e-01 2.36477783744783759400931444361049e-01 3.12879060401825576792633754052986e-03
            3.30541501567599038668987532219035e-01 4.32980714687617229685656639048830e-01 3.12879060401825576792633754052986e-03
            2.36477783744783759400931444361049e-01 3.30541501567599038668987532219035e-01 3.12879060401825576792633754052986e-03
            2.36477783744783759400931444361049e-01 4.32980714687617229685656639048830e-01 3.12879060401825576792633754052986e-03
            4.32980714687617229685656639048830e-01 3.30541501567599038668987532219035e-01 3.12879060401825576792633754052986e-03
            4.32980714687617229685656639048830e-01 2.36477783744783759400931444361049e-01 3.12879060401825576792633754052986e-03
            1.75074013252293603226661389271612e-01 1.22809018053712887263984043784149e-01 1.89062025704546821797125133457484e-03
            1.75074013252293603226661389271612e-01 7.02116968693993426242627720057499e-01 1.89062025704546821797125133457484e-03
            1.22809018053712887263984043784149e-01 1.75074013252293603226661389271612e-01 1.89062025704546821797125133457484e-03
            1.22809018053712887263984043784149e-01 7.02116968693993426242627720057499e-01 1.89062025704546821797125133457484e-03
            7.02116968693993426242627720057499e-01 1.75074013252293603226661389271612e-01 1.89062025704546821797125133457484e-03
            7.02116968693993426242627720057499e-01 1.22809018053712887263984043784149e-01 1.89062025704546821797125133457484e-03
            1.13394973029422363453022626345046e-01 5.50050284444838430419899566459208e-02 1.15318098238608541736271995148400e-03
            1.13394973029422363453022626345046e-01 8.31599998526093786566093513101805e-01 1.15318098238608541736271995148400e-03
            5.50050284444838430419899566459208e-02 1.13394973029422363453022626345046e-01 1.15318098238608541736271995148400e-03
            5.50050284444838430419899566459208e-02 8.31599998526093786566093513101805e-01 1.15318098238608541736271995148400e-03
            8.31599998526093786566093513101805e-01 1.13394973029422363453022626345046e-01 1.15318098238608541736271995148400e-03
            8.31599998526093786566093513101805e-01 5.50050284444838430419899566459208e-02 1.15318098238608541736271995148400e-03
            9.45442284064488130956860345577297e-02 2.87935332271121169345207135847886e-02 7.98215168649472981560188955540980e-04
            9.45442284064488130956860345577297e-02 8.76662238366439083847581059671938e-01 7.98215168649472981560188955540980e-04
            2.87935332271121169345207135847886e-02 9.45442284064488130956860345577297e-02 7.98215168649472981560188955540980e-04
            2.87935332271121169345207135847886e-02 8.76662238366439083847581059671938e-01 7.98215168649472981560188955540980e-04
            8.76662238366439083847581059671938e-01 9.45442284064488130956860345577297e-02 7.98215168649472981560188955540980e-04
            8.76662238366439083847581059671938e-01 2.87935332271121169345207135847886e-02 7.98215168649472981560188955540980e-04
            5.89589423608462975812116724227963e-02 2.62933317918677932067961222628583e-02 6.24418006796972771890730236066247e-04
            5.89589423608462975812116724227963e-02 9.14747725847285919620333061175188e-01 6.24418006796972771890730236066247e-04
            2.62933317918677932067961222628583e-02 5.89589423608462975812116724227963e-02 6.24418006796972771890730236066247e-04
            2.62933317918677932067961222628583e-02 9.14747725847285919620333061175188e-01 6.24418006796972771890730236066247e-04
            9.14747725847285919620333061175188e-01 5.89589423608462975812116724227963e-02 6.24418006796972771890730236066247e-04
            9.14747725847285919620333061175188e-01 2.62933317918677932067961222628583e-02 6.24418006796972771890730236066247e-04
            3.55446117102369674167761104399688e-01 2.83073273852235927916520807912093e-02 1.27781420414727238388241659095002e-03
            3.55446117102369674167761104399688e-01 6.16246555512406701815564247226575e-01 1.27781420414727238388241659095002e-03
            2.83073273852235927916520807912093e-02 3.55446117102369674167761104399688e-01 1.27781420414727238388241659095002e-03
            2.83073273852235927916520807912093e-02 6.16246555512406701815564247226575e-01 1.27781420414727238388241659095002e-03
            6.16246555512406701815564247226575e-01 3.55446117102369674167761104399688e-01 1.27781420414727238388241659095002e-03
            6.16246555512406701815564247226575e-01 2.83073273852235927916520807912093e-02 1.27781420414727238388241659095002e-03
            4.20627844445428888953131263406249e-01 2.81269388071388806715766861543671e-02 1.33914246584528138367398142349884e-03
            4.20627844445428888953131263406249e-01 5.51245216747432209558610338717699e-01 1.33914246584528138367398142349884e-03
            2.81269388071388806715766861543671e-02 4.20627844445428888953131263406249e-01 1.33914246584528138367398142349884e-03
            2.81269388071388806715766861543671e-02 5.51245216747432209558610338717699e-01 1.33914246584528138367398142349884e-03
            5.51245216747432209558610338717699e-01 4.20627844445428888953131263406249e-01 1.33914246584528138367398142349884e-03
            5.51245216747432209558610338717699e-01 2.81269388071388806715766861543671e-02 1.33914246584528138367398142349884e-03
            2.76342477166445721348253528049099e-01 2.10090109306926720728059265752563e-03 3.20695007916894589145712091138307e-04
            2.76342477166445721348253528049099e-01 7.21556621740484960270123337977566e-01 3.20695007916894589145712091138307e-04
            2.10090109306926720728059265752563e-03 2.76342477166445721348253528049099e-01 3.20695007916894589145712091138307e-04
            2.10090109306926720728059265752563e-03 7.21556621740484960270123337977566e-01 3.20695007916894589145712091138307e-04
            7.21556621740484960270123337977566e-01 2.76342477166445721348253528049099e-01 3.20695007916894589145712091138307e-04
            7.21556621740484960270123337977566e-01 2.10090109306926720728059265752563e-03 3.20695007916894589145712091138307e-04
            1.35834405138137270796505617909133e-01 3.05221004022598642324570761275027e-02 9.55358677029271522590514464923217e-04
            1.35834405138137270796505617909133e-01 8.33643494459602840684908642288065e-01 9.55358677029271522590514464923217e-04
            3.05221004022598642324570761275027e-02 1.35834405138137270796505617909133e-01 9.55358677029271522590514464923217e-04
            3.05221004022598642324570761275027e-02 8.33643494459602840684908642288065e-01 9.55358677029271522590514464923217e-04
            8.33643494459602840684908642288065e-01 1.35834405138137270796505617909133e-01 9.55358677029271522590514464923217e-04
            8.33643494459602840684908642288065e-01 3.05221004022598642324570761275027e-02 9.55358677029271522590514464923217e-04
            1.21432444920387294651220599916996e-01 8.81430419815303345387746958294883e-02 1.51428242832449548756212909239594e-03
            1.21432444920387294651220599916996e-01 7.90424513098082370810004704253515e-01 1.51428242832449548756212909239594e-03
            8.81430419815303345387746958294883e-02 1.21432444920387294651220599916996e-01 1.51428242832449548756212909239594e-03
            8.81430419815303345387746958294883e-02 7.90424513098082370810004704253515e-01 1.51428242832449548756212909239594e-03
            7.90424513098082370810004704253515e-01 1.21432444920387294651220599916996e-01 1.51428242832449548756212909239594e-03
            7.90424513098082370810004704253515e-01 8.81430419815303345387746958294883e-02 1.51428242832449548756212909239594e-03
            1.29254481332948201188770553926588e-01 1.25808072722038699620128454625956e-02 5.97009737256886090922547438708534e-04
            1.29254481332948201188770553926588e-01 8.58164711394847956604792216239730e-01 5.97009737256886090922547438708534e-04
            1.25808072722038699620128454625956e-02 1.29254481332948201188770553926588e-01 5.97009737256886090922547438708534e-04
            1.25808072722038699620128454625956e-02 8.58164711394847956604792216239730e-01 5.97009737256886090922547438708534e-04
            8.58164711394847956604792216239730e-01 1.29254481332948201188770553926588e-01 5.97009737256886090922547438708534e-04
            8.58164711394847956604792216239730e-01 1.25808072722038699620128454625956e-02 5.97009737256886090922547438708534e-04
            2.72939318330326308181810190944816e-01 8.38760556578321686549415403533203e-02 1.99489307211005943015758035130602e-03
            2.72939318330326308181810190944816e-01 6.43184626011841453774309229629580e-01 1.99489307211005943015758035130602e-03
            8.38760556578321686549415403533203e-02 2.72939318330326308181810190944816e-01 1.99489307211005943015758035130602e-03
            8.38760556578321686549415403533203e-02 6.43184626011841453774309229629580e-01 1.99489307211005943015758035130602e-03
            6.43184626011841453774309229629580e-01 2.72939318330326308181810190944816e-01 1.99489307211005943015758035130602e-03
            6.43184626011841453774309229629580e-01 8.38760556578321686549415403533203e-02 1.99489307211005943015758035130602e-03
            2.76224431674286952098640313124633e-01 1.24791600444704789407701639447623e-01 2.38707993822377369588783047049674e-03
            2.76224431674286952098640313124633e-01 5.98983967881008161349143392726546e-01 2.38707993822377369588783047049674e-03
            1.24791600444704789407701639447623e-01 2.76224431674286952098640313124633e-01 2.38707993822377369588783047049674e-03
            1.24791600444704789407701639447623e-01 5.98983967881008161349143392726546e-01 2.38707993822377369588783047049674e-03
            5.98983967881008161349143392726546e-01 2.76224431674286952098640313124633e-01 2.38707993822377369588783047049674e-03
            5.98983967881008161349143392726546e-01 1.24791600444704789407701639447623e-01 2.38707993822377369588783047049674e-03
            1.66531061991801065946816606810899e-01 2.23024928943949115808798033810945e-03 2.84050648765331867958566780529850e-04
            1.66531061991801065946816606810899e-01 8.31238688718759499707289251091424e-01 2.84050648765331867958566780529850e-04
            2.23024928943949115808798033810945e-03 1.66531061991801065946816606810899e-01 2.84050648765331867958566780529850e-04
            2.23024928943949115808798033810945e-03 8.31238688718759499707289251091424e-01 2.84050648765331867958566780529850e-04
            8.31238688718759499707289251091424e-01 1.66531061991801065946816606810899e-01 2.84050648765331867958566780529850e-04
            8.31238688718759499707289251091424e-01 2.23024928943949115808798033810945e-03 2.84050648765331867958566780529850e-04
            3.37323861878955577608962812519167e-01 2.28466711100102455242466525930922e-03 3.64644894945768060449259317934434e-04
            3.37323861878955577608962812519167e-01 6.60391471010043362710462133691180e-01 3.64644894945768060449259317934434e-04
            2.28466711100102455242466525930922e-03 3.37323861878955577608962812519167e-01 3.64644894945768060449259317934434e-04
            2.28466711100102455242466525930922e-03 6.60391471010043362710462133691180e-01 3.64644894945768060449259317934434e-04
            6.60391471010043362710462133691180e-01 3.37323861878955577608962812519167e-01 3.64644894945768060449259317934434e-04
            6.60391471010043362710462133691180e-01 2.28466711100102455242466525930922e-03 3.64644894945768060449259317934434e-04
            3.89984837680378781943346666594152e-01 8.31933025800658670911857939245238e-02 2.19977808499303159953108810498179e-03
            3.89984837680378781943346666594152e-01 5.26821859739555420354406578553608e-01 2.19977808499303159953108810498179e-03
            8.31933025800658670911857939245238e-02 3.89984837680378781943346666594152e-01 2.19977808499303159953108810498179e-03
            8.31933025800658670911857939245238e-02 5.26821859739555420354406578553608e-01 2.19977808499303159953108810498179e-03
            5.26821859739555420354406578553608e-01 3.89984837680378781943346666594152e-01 2.19977808499303159953108810498179e-03
            5.26821859739555420354406578553608e-01 8.31933025800658670911857939245238e-02 2.19977808499303159953108810498179e-03
            2.19057402029606912430281795423070e-01 2.40703109183256977465781289993174e-03 3.39479499288667817182701957889890e-04
            2.19057402029606912430281795423070e-01 7.78535566878560514325613439723384e-01 3.39479499288667817182701957889890e-04
            2.40703109183256977465781289993174e-03 2.19057402029606912430281795423070e-01 3.39479499288667817182701957889890e-04
            2.40703109183256977465781289993174e-03 7.78535566878560514325613439723384e-01 3.39479499288667817182701957889890e-04
            7.78535566878560514325613439723384e-01 2.19057402029606912430281795423070e-01 3.39479499288667817182701957889890e-04
            7.78535566878560514325613439723384e-01 2.40703109183256977465781289993174e-03 3.39479499288667817182701957889890e-04
            1.20341342443090318359288914962235e-01 2.41706361847834373668408858293333e-03 2.65722056703090927370913831850885e-04
            1.20341342443090318359288914962235e-01 8.77241593938431263310917529452126e-01 2.65722056703090927370913831850885e-04
            2.41706361847834373668408858293333e-03 1.20341342443090318359288914962235e-01 2.65722056703090927370913831850885e-04
            2.41706361847834373668408858293333e-03 8.77241593938431263310917529452126e-01 2.65722056703090927370913831850885e-04
            8.77241593938431263310917529452126e-01 1.20341342443090318359288914962235e-01 2.65722056703090927370913831850885e-04
            8.77241593938431263310917529452126e-01 2.41706361847834373668408858293333e-03 2.65722056703090927370913831850885e-04
            3.21434597854267645189452196063939e-02 1.28868718184530495207695821591187e-02 3.23762331706671144769538184249313e-04
            3.21434597854267645189452196063939e-02 9.54969668396120185960285198234487e-01 3.23762331706671144769538184249313e-04
            1.28868718184530495207695821591187e-02 3.21434597854267645189452196063939e-02 3.23762331706671144769538184249313e-04
            1.28868718184530495207695821591187e-02 9.54969668396120185960285198234487e-01 3.23762331706671144769538184249313e-04
            9.54969668396120185960285198234487e-01 3.21434597854267645189452196063939e-02 3.23762331706671144769538184249313e-04
            9.54969668396120185960285198234487e-01 1.28868718184530495207695821591187e-02 3.23762331706671144769538184249313e-04
            2.39783625881974243876371133410430e-02 2.58739871201071761422363159965698e-03 1.30663973001150779656293066111061e-04
            2.39783625881974243876371133410430e-02 9.73434238699791842819308840262238e-01 1.30663973001150779656293066111061e-04
            2.58739871201071761422363159965698e-03 2.39783625881974243876371133410430e-02 1.30663973001150779656293066111061e-04
            2.58739871201071761422363159965698e-03 9.73434238699791842819308840262238e-01 1.30663973001150779656293066111061e-04
            9.73434238699791842819308840262238e-01 2.39783625881974243876371133410430e-02 1.30663973001150779656293066111061e-04
            9.73434238699791842819308840262238e-01 2.58739871201071761422363159965698e-03 1.30663973001150779656293066111061e-04
            ];
        
    case  47
        
        % ALG. DEG.:   47
        % PTS CARD.:  399
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.6e-15
        
        xyw=[
            3.49177504154390971891075423627626e-01 3.49177504154390971891075423627626e-01 2.15732847916676184762962975582923e-03
            3.49177504154390971891075423627626e-01 3.01644991691218056217849152744748e-01 2.15732847916676184762962975582923e-03
            3.01644991691218056217849152744748e-01 3.49177504154390971891075423627626e-01 2.15732847916676184762962975582923e-03
            3.02896544117914334748320470680483e-01 3.02896544117914334748320470680483e-01 2.34958561562595468277803156809114e-03
            3.02896544117914334748320470680483e-01 3.94206911764171330503359058639035e-01 2.34958561562595468277803156809114e-03
            3.94206911764171330503359058639035e-01 3.02896544117914334748320470680483e-01 2.34958561562595468277803156809114e-03
            5.05319024623013812735905503359390e-02 5.05319024623013812735905503359390e-02 6.18246204852673482757485778904538e-04
            5.05319024623013812735905503359390e-02 8.98936195075397237452818899328122e-01 6.18246204852673482757485778904538e-04
            8.98936195075397237452818899328122e-01 5.05319024623013812735905503359390e-02 6.18246204852673482757485778904538e-04
            4.21509392878709365781730866729049e-01 4.21509392878709365781730866729049e-01 2.25799868909655303272909776524102e-03
            4.21509392878709365781730866729049e-01 1.56981214242581268436538266541902e-01 2.25799868909655303272909776524102e-03
            1.56981214242581268436538266541902e-01 4.21509392878709365781730866729049e-01 2.25799868909655303272909776524102e-03
            4.74627024342958780955825659475522e-01 4.74627024342958780955825659475522e-01 1.47224053313713067366041986616665e-03
            4.74627024342958780955825659475522e-01 5.07459513140824380883486810489558e-02 1.47224053313713067366041986616665e-03
            5.07459513140824380883486810489558e-02 4.74627024342958780955825659475522e-01 1.47224053313713067366041986616665e-03
            7.95728420692438376260469112821738e-02 7.95728420692438376260469112821738e-02 1.04150124319736590676466203575501e-03
            7.95728420692438376260469112821738e-02 8.40854315861512269236754946177825e-01 1.04150124319736590676466203575501e-03
            8.40854315861512269236754946177825e-01 7.95728420692438376260469112821738e-02 1.04150124319736590676466203575501e-03
            2.36802291430402538086052999233289e-02 2.36802291430402538086052999233289e-02 3.31503931633387379385863313885352e-04
            2.36802291430402538086052999233289e-02 9.52639541713919513199471111875027e-01 3.31503931633387379385863313885352e-04
            9.52639541713919513199471111875027e-01 2.36802291430402538086052999233289e-02 3.31503931633387379385863313885352e-04
            2.63056435770372187832322197209578e-01 2.63056435770372187832322197209578e-01 2.79912721630154025936887407510767e-03
            2.63056435770372187832322197209578e-01 4.73887128459255624335355605580844e-01 2.79912721630154025936887407510767e-03
            4.73887128459255624335355605580844e-01 2.63056435770372187832322197209578e-01 2.79912721630154025936887407510767e-03
            4.42069397263830543565177322307136e-01 4.42069397263830543565177322307136e-01 2.27590725678494636141646623173074e-03
            4.42069397263830543565177322307136e-01 1.15861205472338912869645355385728e-01 2.27590725678494636141646623173074e-03
            1.15861205472338912869645355385728e-01 4.42069397263830543565177322307136e-01 2.27590725678494636141646623173074e-03
            4.86160107483689762908340981084621e-01 4.86160107483689762908340981084621e-01 1.25364067529043347361261684369538e-03
            4.86160107483689762908340981084621e-01 2.76797850326204741833180378307588e-02 1.25364067529043347361261684369538e-03
            2.76797850326204741833180378307588e-02 4.86160107483689762908340981084621e-01 1.25364067529043347361261684369538e-03
            1.62146122522450308878205760265701e-01 1.62146122522450308878205760265701e-01 2.03221496293360314941423006018795e-03
            1.62146122522450308878205760265701e-01 6.75707754955099382243588479468599e-01 2.03221496293360314941423006018795e-03
            6.75707754955099382243588479468599e-01 1.62146122522450308878205760265701e-01 2.03221496293360314941423006018795e-03
            3.73691181805182015551736185443588e-01 3.73691181805182015551736185443588e-01 2.75255677067570963395537475548736e-03
            3.73691181805182015551736185443588e-01 2.52617636389635968896527629112825e-01 2.75255677067570963395537475548736e-03
            2.52617636389635968896527629112825e-01 3.73691181805182015551736185443588e-01 2.75255677067570963395537475548736e-03
            2.12551268904813739357706481314381e-01 2.12551268904813739357706481314381e-01 2.47233079993487659883899532076157e-03
            2.12551268904813739357706481314381e-01 5.74897462190372521284587037371239e-01 2.47233079993487659883899532076157e-03
            5.74897462190372521284587037371239e-01 2.12551268904813739357706481314381e-01 2.47233079993487659883899532076157e-03
            4.59832174149501116122706889655092e-01 4.59832174149501116122706889655092e-01 2.03289490379708181339024619660449e-03
            4.59832174149501116122706889655092e-01 8.03356517009977677545862206898164e-02 2.03289490379708181339024619660449e-03
            8.03356517009977677545862206898164e-02 4.59832174149501116122706889655092e-01 2.03289490379708181339024619660449e-03
            4.98918529148673406403702301759040e-01 4.98918529148673406403702301759040e-01 3.74415430601990112304683355759494e-04
            4.98918529148673406403702301759040e-01 2.16294170265318719259539648192003e-03 3.74415430601990112304683355759494e-04
            2.16294170265318719259539648192003e-03 4.98918529148673406403702301759040e-01 3.74415430601990112304683355759494e-04
            1.24974506748911293185777537928516e-01 1.24974506748911293185777537928516e-01 1.81606992783187883543083440684995e-03
            1.24974506748911293185777537928516e-01 7.50050986502177385872869308514055e-01 1.81606992783187883543083440684995e-03
            7.50050986502177385872869308514055e-01 1.24974506748911293185777537928516e-01 1.81606992783187883543083440684995e-03
            2.60509669473600602440654938618536e-03 2.60509669473600602440654938618536e-03 4.46268307090296161964565468949218e-05
            2.60509669473600602440654938618536e-03 9.94789806610527960195611285598716e-01 4.46268307090296161964565468949218e-05
            9.94789806610527960195611285598716e-01 2.60509669473600602440654938618536e-03 4.46268307090296161964565468949218e-05
            1.32182347742614610813793873944633e-02 6.92610811899467895824678009830677e-03 1.06391003262472626797810471011019e-04
            1.32182347742614610813793873944633e-02 9.79855657106743938022930251463549e-01 1.06391003262472626797810471011019e-04
            6.92610811899467895824678009830677e-03 1.32182347742614610813793873944633e-02 1.06391003262472626797810471011019e-04
            6.92610811899467895824678009830677e-03 9.79855657106743938022930251463549e-01 1.06391003262472626797810471011019e-04
            9.79855657106743938022930251463549e-01 1.32182347742614610813793873944633e-02 1.06391003262472626797810471011019e-04
            9.79855657106743938022930251463549e-01 6.92610811899467895824678009830677e-03 1.06391003262472626797810471011019e-04
            1.82074143261559212580991129470931e-01 8.16358471109855377001451870455639e-02 1.12574137246430677414266163793855e-03
            1.82074143261559212580991129470931e-01 7.36290009627455166452136836596765e-01 1.12574137246430677414266163793855e-03
            8.16358471109855377001451870455639e-02 1.82074143261559212580991129470931e-01 1.12574137246430677414266163793855e-03
            8.16358471109855377001451870455639e-02 7.36290009627455166452136836596765e-01 1.12574137246430677414266163793855e-03
            7.36290009627455166452136836596765e-01 1.82074143261559212580991129470931e-01 1.12574137246430677414266163793855e-03
            7.36290009627455166452136836596765e-01 8.16358471109855377001451870455639e-02 1.12574137246430677414266163793855e-03
            1.50580927317701368517433024862839e-01 8.61963659887330785336345684299886e-02 1.13253505774401773582849006771767e-03
            1.50580927317701368517433024862839e-01 7.63222706693565622337871445779456e-01 1.13253505774401773582849006771767e-03
            8.61963659887330785336345684299886e-02 1.50580927317701368517433024862839e-01 1.13253505774401773582849006771767e-03
            8.61963659887330785336345684299886e-02 7.63222706693565622337871445779456e-01 1.13253505774401773582849006771767e-03
            7.63222706693565622337871445779456e-01 1.50580927317701368517433024862839e-01 1.13253505774401773582849006771767e-03
            7.63222706693565622337871445779456e-01 8.61963659887330785336345684299886e-02 1.13253505774401773582849006771767e-03
            7.64329826567074904763998688395077e-02 5.02931805301914183758960064096755e-02 7.01364787340123113645873331734037e-04
            7.64329826567074904763998688395077e-02 8.73273836813101111964385836472502e-01 7.01364787340123113645873331734037e-04
            5.02931805301914183758960064096755e-02 7.64329826567074904763998688395077e-02 7.01364787340123113645873331734037e-04
            5.02931805301914183758960064096755e-02 8.73273836813101111964385836472502e-01 7.01364787340123113645873331734037e-04
            8.73273836813101111964385836472502e-01 7.64329826567074904763998688395077e-02 7.01364787340123113645873331734037e-04
            8.73273836813101111964385836472502e-01 5.02931805301914183758960064096755e-02 7.01364787340123113645873331734037e-04
            1.38326160976293081189547606868473e-02 1.04889894991745235425240068138919e-03 4.89049584063299501023210080763448e-05
            1.38326160976293081189547606868473e-02 9.85118484952453243863601528573781e-01 4.89049584063299501023210080763448e-05
            1.04889894991745235425240068138919e-03 1.38326160976293081189547606868473e-02 4.89049584063299501023210080763448e-05
            1.04889894991745235425240068138919e-03 9.85118484952453243863601528573781e-01 4.89049584063299501023210080763448e-05
            9.85118484952453243863601528573781e-01 1.38326160976293081189547606868473e-02 4.89049584063299501023210080763448e-05
            9.85118484952453243863601528573781e-01 1.04889894991745235425240068138919e-03 4.89049584063299501023210080763448e-05
            2.67452272430761042731894860935427e-02 1.00204975653522664752470561211339e-02 2.01931493401746626082521318679142e-04
            2.67452272430761042731894860935427e-02 9.63234275191571565066794846643461e-01 2.01931493401746626082521318679142e-04
            1.00204975653522664752470561211339e-02 2.67452272430761042731894860935427e-02 2.01931493401746626082521318679142e-04
            1.00204975653522664752470561211339e-02 9.63234275191571565066794846643461e-01 2.01931493401746626082521318679142e-04
            9.63234275191571565066794846643461e-01 2.67452272430761042731894860935427e-02 2.01931493401746626082521318679142e-04
            9.63234275191571565066794846643461e-01 1.00204975653522664752470561211339e-02 2.01931493401746626082521318679142e-04
            3.71190236752514501894495424494380e-01 2.03674253045179914911599894367100e-01 2.57682766030653976480069289323183e-03
            3.71190236752514501894495424494380e-01 4.25135510202305555438329065509606e-01 2.57682766030653976480069289323183e-03
            2.03674253045179914911599894367100e-01 3.71190236752514501894495424494380e-01 2.57682766030653976480069289323183e-03
            2.03674253045179914911599894367100e-01 4.25135510202305555438329065509606e-01 2.57682766030653976480069289323183e-03
            4.25135510202305555438329065509606e-01 3.71190236752514501894495424494380e-01 2.57682766030653976480069289323183e-03
            4.25135510202305555438329065509606e-01 2.03674253045179914911599894367100e-01 2.57682766030653976480069289323183e-03
            3.68894972792118358562163393798983e-01 1.59326683873933672153100360446842e-01 2.25473552819196879429264335215066e-03
            3.68894972792118358562163393798983e-01 4.71778343333947969284736245754175e-01 2.25473552819196879429264335215066e-03
            1.59326683873933672153100360446842e-01 3.68894972792118358562163393798983e-01 2.25473552819196879429264335215066e-03
            1.59326683873933672153100360446842e-01 4.71778343333947969284736245754175e-01 2.25473552819196879429264335215066e-03
            4.71778343333947969284736245754175e-01 3.68894972792118358562163393798983e-01 2.25473552819196879429264335215066e-03
            4.71778343333947969284736245754175e-01 1.59326683873933672153100360446842e-01 2.25473552819196879429264335215066e-03
            1.07074579893119942175516712268291e-01 5.33687025646959883884434816536668e-02 9.57294483603012154836198188689877e-04
            1.07074579893119942175516712268291e-01 8.39556717542184083313827613892499e-01 9.57294483603012154836198188689877e-04
            5.33687025646959883884434816536668e-02 1.07074579893119942175516712268291e-01 9.57294483603012154836198188689877e-04
            5.33687025646959883884434816536668e-02 8.39556717542184083313827613892499e-01 9.57294483603012154836198188689877e-04
            8.39556717542184083313827613892499e-01 1.07074579893119942175516712268291e-01 9.57294483603012154836198188689877e-04
            8.39556717542184083313827613892499e-01 5.33687025646959883884434816536668e-02 9.57294483603012154836198188689877e-04
            1.15217912167360214681366414879449e-01 8.52960644923164285069461243438127e-02 1.30313259149965202446375123201960e-03
            1.15217912167360214681366414879449e-01 7.99486023340323370689475268591195e-01 1.30313259149965202446375123201960e-03
            8.52960644923164285069461243438127e-02 1.15217912167360214681366414879449e-01 1.30313259149965202446375123201960e-03
            8.52960644923164285069461243438127e-02 7.99486023340323370689475268591195e-01 1.30313259149965202446375123201960e-03
            7.99486023340323370689475268591195e-01 1.15217912167360214681366414879449e-01 1.30313259149965202446375123201960e-03
            7.99486023340323370689475268591195e-01 8.52960644923164285069461243438127e-02 1.30313259149965202446375123201960e-03
            4.17962853969899517103669950301992e-01 5.14822979290668059082491936351289e-02 1.53807863714332422599984084854441e-03
            4.17962853969899517103669950301992e-01 5.30554848101033704743656471691793e-01 1.53807863714332422599984084854441e-03
            5.14822979290668059082491936351289e-02 4.17962853969899517103669950301992e-01 1.53807863714332422599984084854441e-03
            5.14822979290668059082491936351289e-02 5.30554848101033704743656471691793e-01 1.53807863714332422599984084854441e-03
            5.30554848101033704743656471691793e-01 4.17962853969899517103669950301992e-01 1.53807863714332422599984084854441e-03
            5.30554848101033704743656471691793e-01 5.14822979290668059082491936351289e-02 1.53807863714332422599984084854441e-03
            4.62992599159318674040264340874273e-02 2.75536314394735394606161804631483e-02 5.16406156255945643883398243900729e-04
            4.62992599159318674040264340874273e-02 9.26147108644594596604804337403039e-01 5.16406156255945643883398243900729e-04
            2.75536314394735394606161804631483e-02 4.62992599159318674040264340874273e-02 5.16406156255945643883398243900729e-04
            2.75536314394735394606161804631483e-02 9.26147108644594596604804337403039e-01 5.16406156255945643883398243900729e-04
            9.26147108644594596604804337403039e-01 4.62992599159318674040264340874273e-02 5.16406156255945643883398243900729e-04
            9.26147108644594596604804337403039e-01 2.75536314394735394606161804631483e-02 5.16406156255945643883398243900729e-04
            3.25029937315999439828573258637334e-01 1.18702027168940849044531660183566e-01 2.17718061520642477968401351517969e-03
            3.25029937315999439828573258637334e-01 5.56268035515059655615743849921273e-01 2.17718061520642477968401351517969e-03
            1.18702027168940849044531660183566e-01 3.25029937315999439828573258637334e-01 2.17718061520642477968401351517969e-03
            1.18702027168940849044531660183566e-01 5.56268035515059655615743849921273e-01 2.17718061520642477968401351517969e-03
            5.56268035515059655615743849921273e-01 3.25029937315999439828573258637334e-01 2.17718061520642477968401351517969e-03
            5.56268035515059655615743849921273e-01 1.18702027168940849044531660183566e-01 2.17718061520642477968401351517969e-03
            2.23021456853453786672858427664323e-01 8.50615771696930367262723393650958e-02 1.69604395891152196114426420336940e-03
            2.23021456853453786672858427664323e-01 6.91916965976853148845293617341667e-01 1.69604395891152196114426420336940e-03
            8.50615771696930367262723393650958e-02 2.23021456853453786672858427664323e-01 1.69604395891152196114426420336940e-03
            8.50615771696930367262723393650958e-02 6.91916965976853148845293617341667e-01 1.69604395891152196114426420336940e-03
            6.91916965976853148845293617341667e-01 2.23021456853453786672858427664323e-01 1.69604395891152196114426420336940e-03
            6.91916965976853148845293617341667e-01 8.50615771696930367262723393650958e-02 1.69604395891152196114426420336940e-03
            3.82240080251783131970455542614218e-01 1.18127508441192408117359491370735e-01 2.26657835230176501334353744709915e-03
            3.82240080251783131970455542614218e-01 4.99632411307024459912184966015047e-01 2.26657835230176501334353744709915e-03
            1.18127508441192408117359491370735e-01 3.82240080251783131970455542614218e-01 2.26657835230176501334353744709915e-03
            1.18127508441192408117359491370735e-01 4.99632411307024459912184966015047e-01 2.26657835230176501334353744709915e-03
            4.99632411307024459912184966015047e-01 3.82240080251783131970455542614218e-01 2.26657835230176501334353744709915e-03
            4.99632411307024459912184966015047e-01 1.18127508441192408117359491370735e-01 2.26657835230176501334353744709915e-03
            3.18681016804057470892530545825139e-01 2.52708350652741664532641152618453e-01 2.68485965378233984099387576804929e-03
            3.18681016804057470892530545825139e-01 4.28610632543200864574828301556408e-01 2.68485965378233984099387576804929e-03
            2.52708350652741664532641152618453e-01 3.18681016804057470892530545825139e-01 2.68485965378233984099387576804929e-03
            2.52708350652741664532641152618453e-01 4.28610632543200864574828301556408e-01 2.68485965378233984099387576804929e-03
            4.28610632543200864574828301556408e-01 3.18681016804057470892530545825139e-01 2.68485965378233984099387576804929e-03
            4.28610632543200864574828301556408e-01 2.52708350652741664532641152618453e-01 2.68485965378233984099387576804929e-03
            3.16102429197291978635320219837013e-01 2.05745941208171606584542701057217e-01 2.50562594513383668504147294697759e-03
            3.16102429197291978635320219837013e-01 4.78151629594536387024561463476857e-01 2.50562594513383668504147294697759e-03
            2.05745941208171606584542701057217e-01 3.16102429197291978635320219837013e-01 2.50562594513383668504147294697759e-03
            2.05745941208171606584542701057217e-01 4.78151629594536387024561463476857e-01 2.50562594513383668504147294697759e-03
            4.78151629594536387024561463476857e-01 3.16102429197291978635320219837013e-01 2.50562594513383668504147294697759e-03
            4.78151629594536387024561463476857e-01 2.05745941208171606584542701057217e-01 2.50562594513383668504147294697759e-03
            2.01714360262501668508150487468811e-01 2.86457371054149485811723963024633e-02 9.76697137394438386417960806795691e-04
            2.01714360262501668508150487468811e-01 7.69639902632083372502336260367883e-01 9.76697137394438386417960806795691e-04
            2.86457371054149485811723963024633e-02 2.01714360262501668508150487468811e-01 9.76697137394438386417960806795691e-04
            2.86457371054149485811723963024633e-02 7.69639902632083372502336260367883e-01 9.76697137394438386417960806795691e-04
            7.69639902632083372502336260367883e-01 2.01714360262501668508150487468811e-01 9.76697137394438386417960806795691e-04
            7.69639902632083372502336260367883e-01 2.86457371054149485811723963024633e-02 9.76697137394438386417960806795691e-04
            3.04095374457296840908782087353757e-01 2.87008250916218343729013895426760e-02 1.13507664579970629983862728806798e-03
            3.04095374457296840908782087353757e-01 6.67203800451081296962740907474654e-01 1.13507664579970629983862728806798e-03
            2.87008250916218343729013895426760e-02 3.04095374457296840908782087353757e-01 1.13507664579970629983862728806798e-03
            2.87008250916218343729013895426760e-02 6.67203800451081296962740907474654e-01 1.13507664579970629983862728806798e-03
            6.67203800451081296962740907474654e-01 3.04095374457296840908782087353757e-01 1.13507664579970629983862728806798e-03
            6.67203800451081296962740907474654e-01 2.87008250916218343729013895426760e-02 1.13507664579970629983862728806798e-03
            2.50940600094452903601194293514709e-01 2.90822358487935334059315550803149e-02 1.06933649152424442400899362581868e-03
            2.50940600094452903601194293514709e-01 7.19977164056753493603935112332692e-01 1.06933649152424442400899362581868e-03
            2.90822358487935334059315550803149e-02 2.50940600094452903601194293514709e-01 1.06933649152424442400899362581868e-03
            2.90822358487935334059315550803149e-02 7.19977164056753493603935112332692e-01 1.06933649152424442400899362581868e-03
            7.19977164056753493603935112332692e-01 2.50940600094452903601194293514709e-01 1.06933649152424442400899362581868e-03
            7.19977164056753493603935112332692e-01 2.90822358487935334059315550803149e-02 1.06933649152424442400899362581868e-03
            2.70113685487746946201070841198089e-01 1.21742235940660512016897598641663e-01 2.13906709163620830857488464005201e-03
            2.70113685487746946201070841198089e-01 6.08144078571592583415394983603619e-01 2.13906709163620830857488464005201e-03
            1.21742235940660512016897598641663e-01 2.70113685487746946201070841198089e-01 2.13906709163620830857488464005201e-03
            1.21742235940660512016897598641663e-01 6.08144078571592583415394983603619e-01 2.13906709163620830857488464005201e-03
            6.08144078571592583415394983603619e-01 2.70113685487746946201070841198089e-01 2.13906709163620830857488464005201e-03
            6.08144078571592583415394983603619e-01 1.21742235940660512016897598641663e-01 2.13906709163620830857488464005201e-03
            4.23007886771842922524200503175962e-01 2.79469184369985149118953415836586e-02 1.25350967222140428219967489553710e-03
            4.23007886771842922524200503175962e-01 5.49045194791158541747222443518694e-01 1.25350967222140428219967489553710e-03
            2.79469184369985149118953415836586e-02 4.23007886771842922524200503175962e-01 1.25350967222140428219967489553710e-03
            2.79469184369985149118953415836586e-02 5.49045194791158541747222443518694e-01 1.25350967222140428219967489553710e-03
            5.49045194791158541747222443518694e-01 4.23007886771842922524200503175962e-01 1.25350967222140428219967489553710e-03
            5.49045194791158541747222443518694e-01 2.79469184369985149118953415836586e-02 1.25350967222140428219967489553710e-03
            2.18838108408261983184317500672478e-01 1.22140066758372467514703885171912e-01 1.98686555019924725867808135149062e-03
            2.18838108408261983184317500672478e-01 6.59021824833365466034251767268870e-01 1.98686555019924725867808135149062e-03
            1.22140066758372467514703885171912e-01 2.18838108408261983184317500672478e-01 1.98686555019924725867808135149062e-03
            1.22140066758372467514703885171912e-01 6.59021824833365466034251767268870e-01 1.98686555019924725867808135149062e-03
            6.59021824833365466034251767268870e-01 2.18838108408261983184317500672478e-01 1.98686555019924725867808135149062e-03
            6.59021824833365466034251767268870e-01 1.22140066758372467514703885171912e-01 1.98686555019924725867808135149062e-03
            3.15558282430156267217569165950408e-01 1.60412985091024695716299675041228e-01 2.37020684316891329726506576491829e-03
            3.15558282430156267217569165950408e-01 5.24028732478819092577282390266191e-01 2.37020684316891329726506576491829e-03
            1.60412985091024695716299675041228e-01 3.15558282430156267217569165950408e-01 2.37020684316891329726506576491829e-03
            1.60412985091024695716299675041228e-01 5.24028732478819092577282390266191e-01 2.37020684316891329726506576491829e-03
            5.24028732478819092577282390266191e-01 3.15558282430156267217569165950408e-01 2.37020684316891329726506576491829e-03
            5.24028732478819092577282390266191e-01 1.60412985091024695716299675041228e-01 2.37020684316891329726506576491829e-03
            3.59994205076786699670776670245687e-01 5.20384051030055791442485713105270e-02 1.58045107889224069157296792553780e-03
            3.59994205076786699670776670245687e-01 5.87967389820207686490505238907645e-01 1.58045107889224069157296792553780e-03
            5.20384051030055791442485713105270e-02 3.59994205076786699670776670245687e-01 1.58045107889224069157296792553780e-03
            5.20384051030055791442485713105270e-02 5.87967389820207686490505238907645e-01 1.58045107889224069157296792553780e-03
            5.87967389820207686490505238907645e-01 3.59994205076786699670776670245687e-01 1.58045107889224069157296792553780e-03
            5.87967389820207686490505238907645e-01 5.20384051030055791442485713105270e-02 1.58045107889224069157296792553780e-03
            7.61492805627096702725253862809041e-02 2.80080973264347880824498560059510e-02 6.53246515891860494128218217468884e-04
            7.61492805627096702725253862809041e-02 8.95842622110855524297789997945074e-01 6.53246515891860494128218217468884e-04
            2.80080973264347880824498560059510e-02 7.61492805627096702725253862809041e-02 6.53246515891860494128218217468884e-04
            2.80080973264347880824498560059510e-02 8.95842622110855524297789997945074e-01 6.53246515891860494128218217468884e-04
            8.95842622110855524297789997945074e-01 7.61492805627096702725253862809041e-02 6.53246515891860494128218217468884e-04
            8.95842622110855524297789997945074e-01 2.80080973264347880824498560059510e-02 6.53246515891860494128218217468884e-04
            3.61687125042856494339105211111018e-01 2.83641992932530291604553696060975e-02 1.21120828529704105923359147567453e-03
            3.61687125042856494339105211111018e-01 6.09948675663890438336522947793128e-01 1.21120828529704105923359147567453e-03
            2.83641992932530291604553696060975e-02 3.61687125042856494339105211111018e-01 1.21120828529704105923359147567453e-03
            2.83641992932530291604553696060975e-02 6.09948675663890438336522947793128e-01 1.21120828529704105923359147567453e-03
            6.09948675663890438336522947793128e-01 3.61687125042856494339105211111018e-01 1.21120828529704105923359147567453e-03
            6.09948675663890438336522947793128e-01 2.83641992932530291604553696060975e-02 1.21120828529704105923359147567453e-03
            3.96957042979479091115990740945563e-01 8.18149550216045046147073094289226e-02 2.06687929519422559920815984924047e-03
            3.96957042979479091115990740945563e-01 5.21228001998916390391514141811058e-01 2.06687929519422559920815984924047e-03
            8.18149550216045046147073094289226e-02 3.96957042979479091115990740945563e-01 2.06687929519422559920815984924047e-03
            8.18149550216045046147073094289226e-02 5.21228001998916390391514141811058e-01 2.06687929519422559920815984924047e-03
            5.21228001998916390391514141811058e-01 3.96957042979479091115990740945563e-01 2.06687929519422559920815984924047e-03
            5.21228001998916390391514141811058e-01 8.18149550216045046147073094289226e-02 2.06687929519422559920815984924047e-03
            2.63582104650385729005535040414543e-01 2.12133896316561210149131966318237e-01 2.55256659060561609878026168019005e-03
            2.63582104650385729005535040414543e-01 5.24283999033053116356484224525047e-01 2.55256659060561609878026168019005e-03
            2.12133896316561210149131966318237e-01 2.63582104650385729005535040414543e-01 2.55256659060561609878026168019005e-03
            2.12133896316561210149131966318237e-01 5.24283999033053116356484224525047e-01 2.55256659060561609878026168019005e-03
            5.24283999033053116356484224525047e-01 2.63582104650385729005535040414543e-01 2.55256659060561609878026168019005e-03
            5.24283999033053116356484224525047e-01 2.12133896316561210149131966318237e-01 2.55256659060561609878026168019005e-03
            1.92535054212271156126590199164639e-01 5.24452042229162709863210523053567e-02 1.34881166699170914995142833703312e-03
            1.92535054212271156126590199164639e-01 7.55019741564812552070407036808319e-01 1.34881166699170914995142833703312e-03
            5.24452042229162709863210523053567e-02 1.92535054212271156126590199164639e-01 1.34881166699170914995142833703312e-03
            5.24452042229162709863210523053567e-02 7.55019741564812552070407036808319e-01 1.34881166699170914995142833703312e-03
            7.55019741564812552070407036808319e-01 1.92535054212271156126590199164639e-01 1.34881166699170914995142833703312e-03
            7.55019741564812552070407036808319e-01 5.24452042229162709863210523053567e-02 1.34881166699170914995142833703312e-03
            1.55053798001932918726453181079705e-01 2.91776764327326132431039695802610e-02 9.40482198953650075715082756744323e-04
            1.55053798001932918726453181079705e-01 8.15768525565334523541594080597861e-01 9.40482198953650075715082756744323e-04
            2.91776764327326132431039695802610e-02 1.55053798001932918726453181079705e-01 9.40482198953650075715082756744323e-04
            2.91776764327326132431039695802610e-02 8.15768525565334523541594080597861e-01 9.40482198953650075715082756744323e-04
            8.15768525565334523541594080597861e-01 1.55053798001932918726453181079705e-01 9.40482198953650075715082756744323e-04
            8.15768525565334523541594080597861e-01 2.91776764327326132431039695802610e-02 9.40482198953650075715082756744323e-04
            8.08139183670610050658211775953532e-02 1.16752954894914842687292377831909e-02 4.57109505137131535892208722415830e-04
            8.08139183670610050658211775953532e-02 9.07510786143447489848767872899771e-01 4.57109505137131535892208722415830e-04
            1.16752954894914842687292377831909e-02 8.08139183670610050658211775953532e-02 4.57109505137131535892208722415830e-04
            1.16752954894914842687292377831909e-02 9.07510786143447489848767872899771e-01 4.57109505137131535892208722415830e-04
            9.07510786143447489848767872899771e-01 8.08139183670610050658211775953532e-02 4.57109505137131535892208722415830e-04
            9.07510786143447489848767872899771e-01 1.16752954894914842687292377831909e-02 4.57109505137131535892208722415830e-04
            4.94903005695827707954848051485897e-02 1.13529551848956314352712126947154e-02 3.50532801434540880469725721013674e-04
            4.94903005695827707954848051485897e-02 9.39156744245521535319198846991640e-01 3.50532801434540880469725721013674e-04
            1.13529551848956314352712126947154e-02 4.94903005695827707954848051485897e-02 3.50532801434540880469725721013674e-04
            1.13529551848956314352712126947154e-02 9.39156744245521535319198846991640e-01 3.50532801434540880469725721013674e-04
            9.39156744245521535319198846991640e-01 4.94903005695827707954848051485897e-02 3.50532801434540880469725721013674e-04
            9.39156744245521535319198846991640e-01 1.13529551848956314352712126947154e-02 3.50532801434540880469725721013674e-04
            3.01366119175060842572122510318877e-01 5.27412086548005240582703834206768e-02 1.59505282435752713393606327940688e-03
            3.01366119175060842572122510318877e-01 6.45892672170138681941864433611045e-01 1.59505282435752713393606327940688e-03
            5.27412086548005240582703834206768e-02 3.01366119175060842572122510318877e-01 1.59505282435752713393606327940688e-03
            5.27412086548005240582703834206768e-02 6.45892672170138681941864433611045e-01 1.59505282435752713393606327940688e-03
            6.45892672170138681941864433611045e-01 3.01366119175060842572122510318877e-01 1.59505282435752713393606327940688e-03
            6.45892672170138681941864433611045e-01 5.27412086548005240582703834206768e-02 1.59505282435752713393606327940688e-03
            1.45559094124306370732213622432027e-01 5.39363844971338171618846502042288e-02 1.21691292055276802618057896410164e-03
            1.45559094124306370732213622432027e-01 8.00504521378559874555946862528799e-01 1.21691292055276802618057896410164e-03
            5.39363844971338171618846502042288e-02 1.45559094124306370732213622432027e-01 1.21691292055276802618057896410164e-03
            5.39363844971338171618846502042288e-02 8.00504521378559874555946862528799e-01 1.21691292055276802618057896410164e-03
            8.00504521378559874555946862528799e-01 1.45559094124306370732213622432027e-01 1.21691292055276802618057896410164e-03
            8.00504521378559874555946862528799e-01 5.39363844971338171618846502042288e-02 1.21691292055276802618057896410164e-03
            3.39239311459584025332247847472900e-02 2.02689658380067255424861194512687e-03 1.25423725728734887551255039461751e-04
            3.39239311459584025332247847472900e-02 9.64049172270241006010849105223315e-01 1.25423725728734887551255039461751e-04
            2.02689658380067255424861194512687e-03 3.39239311459584025332247847472900e-02 1.25423725728734887551255039461751e-04
            2.02689658380067255424861194512687e-03 9.64049172270241006010849105223315e-01 1.25423725728734887551255039461751e-04
            9.64049172270241006010849105223315e-01 3.39239311459584025332247847472900e-02 1.25423725728734887551255039461751e-04
            9.64049172270241006010849105223315e-01 2.02689658380067255424861194512687e-03 1.25423725728734887551255039461751e-04
            4.60920398298352129717869729574886e-01 1.13741139787508632741364067442191e-02 8.55406563880253541058695798682265e-04
            4.60920398298352129717869729574886e-01 5.27705487722897048641357287124265e-01 8.55406563880253541058695798682265e-04
            1.13741139787508632741364067442191e-02 4.60920398298352129717869729574886e-01 8.55406563880253541058695798682265e-04
            1.13741139787508632741364067442191e-02 5.27705487722897048641357287124265e-01 8.55406563880253541058695798682265e-04
            5.27705487722897048641357287124265e-01 4.60920398298352129717869729574886e-01 8.55406563880253541058695798682265e-04
            5.27705487722897048641357287124265e-01 1.13741139787508632741364067442191e-02 8.55406563880253541058695798682265e-04
            2.09650636992742611663942398081417e-01 1.64719735798059069287191391595115e-01 2.26400606671470303737914164798894e-03
            2.09650636992742611663942398081417e-01 6.25629627209198235782139363436727e-01 2.26400606671470303737914164798894e-03
            1.64719735798059069287191391595115e-01 2.09650636992742611663942398081417e-01 2.26400606671470303737914164798894e-03
            1.64719735798059069287191391595115e-01 6.25629627209198235782139363436727e-01 2.26400606671470303737914164798894e-03
            6.25629627209198235782139363436727e-01 2.09650636992742611663942398081417e-01 2.26400606671470303737914164798894e-03
            6.25629627209198235782139363436727e-01 1.64719735798059069287191391595115e-01 2.26400606671470303737914164798894e-03
            3.35618830727636319188889046927216e-01 8.24852827778116137746877711833804e-02 2.00406961507452368101422912616272e-03
            3.35618830727636319188889046927216e-01 5.81895886494552039280847566260491e-01 2.00406961507452368101422912616272e-03
            8.24852827778116137746877711833804e-02 3.35618830727636319188889046927216e-01 2.00406961507452368101422912616272e-03
            8.24852827778116137746877711833804e-02 5.81895886494552039280847566260491e-01 2.00406961507452368101422912616272e-03
            5.81895886494552039280847566260491e-01 3.35618830727636319188889046927216e-01 2.00406961507452368101422912616272e-03
            5.81895886494552039280847566260491e-01 8.24852827778116137746877711833804e-02 2.00406961507452368101422912616272e-03
            1.19198356094802629390194681491266e-01 1.18660258094383762317303165900739e-02 5.59770199953665294721749656758902e-04
            1.19198356094802629390194681491266e-01 8.68935618095758921519689010892762e-01 5.59770199953665294721749656758902e-04
            1.18660258094383762317303165900739e-02 1.19198356094802629390194681491266e-01 5.59770199953665294721749656758902e-04
            1.18660258094383762317303165900739e-02 8.68935618095758921519689010892762e-01 5.59770199953665294721749656758902e-04
            8.68935618095758921519689010892762e-01 1.19198356094802629390194681491266e-01 5.59770199953665294721749656758902e-04
            8.68935618095758921519689010892762e-01 1.18660258094383762317303165900739e-02 5.59770199953665294721749656758902e-04
            1.12751742717588893105684633155761e-01 2.89885801232179704522984309278399e-02 8.27824540613835498266415147128328e-04
            1.12751742717588893105684633155761e-01 8.58259677159193157258698647638084e-01 8.27824540613835498266415147128328e-04
            2.89885801232179704522984309278399e-02 1.12751742717588893105684633155761e-01 8.27824540613835498266415147128328e-04
            2.89885801232179704522984309278399e-02 8.58259677159193157258698647638084e-01 8.27824540613835498266415147128328e-04
            8.58259677159193157258698647638084e-01 1.12751742717588893105684633155761e-01 8.27824540613835498266415147128328e-04
            8.58259677159193157258698647638084e-01 2.89885801232179704522984309278399e-02 8.27824540613835498266415147128328e-04
            2.44604484993710485918327890431101e-01 5.36092519706683826030690909192344e-02 1.53727860563678919540719913783278e-03
            2.44604484993710485918327890431101e-01 7.01786263035621193928648153814720e-01 1.53727860563678919540719913783278e-03
            5.36092519706683826030690909192344e-02 2.44604484993710485918327890431101e-01 1.53727860563678919540719913783278e-03
            5.36092519706683826030690909192344e-02 7.01786263035621193928648153814720e-01 1.53727860563678919540719913783278e-03
            7.01786263035621193928648153814720e-01 2.44604484993710485918327890431101e-01 1.53727860563678919540719913783278e-03
            7.01786263035621193928648153814720e-01 5.36092519706683826030690909192344e-02 1.53727860563678919540719913783278e-03
            2.61327895213015415354362858124659e-01 1.64967292586388120101403842454602e-01 2.41503521209857595353209802624406e-03
            2.61327895213015415354362858124659e-01 5.73704812200596547810960146307480e-01 2.41503521209857595353209802624406e-03
            1.64967292586388120101403842454602e-01 2.61327895213015415354362858124659e-01 2.41503521209857595353209802624406e-03
            1.64967292586388120101403842454602e-01 5.73704812200596547810960146307480e-01 2.41503521209857595353209802624406e-03
            5.73704812200596547810960146307480e-01 2.61327895213015415354362858124659e-01 2.41503521209857595353209802624406e-03
            5.73704812200596547810960146307480e-01 1.64967292586388120101403842454602e-01 2.41503521209857595353209802624406e-03
            2.76710222358847013346405674383277e-01 8.41334586353880142794992025301326e-02 1.94934870191071406633764873106429e-03
            2.76710222358847013346405674383277e-01 6.39156319005764972374095123086590e-01 1.94934870191071406633764873106429e-03
            8.41334586353880142794992025301326e-02 2.76710222358847013346405674383277e-01 1.94934870191071406633764873106429e-03
            8.41334586353880142794992025301326e-02 6.39156319005764972374095123086590e-01 1.94934870191071406633764873106429e-03
            6.39156319005764972374095123086590e-01 2.76710222358847013346405674383277e-01 1.94934870191071406633764873106429e-03
            6.39156319005764972374095123086590e-01 8.41334586353880142794992025301326e-02 1.94934870191071406633764873106429e-03
            3.31057698305513958825230247384752e-01 1.17181407852650371764768877369534e-02 8.22925995577372887748723861989220e-04
            3.31057698305513958825230247384752e-01 6.57224160909220977977440725226188e-01 8.22925995577372887748723861989220e-04
            1.17181407852650371764768877369534e-02 3.31057698305513958825230247384752e-01 8.22925995577372887748723861989220e-04
            1.17181407852650371764768877369534e-02 6.57224160909220977977440725226188e-01 8.22925995577372887748723861989220e-04
            6.57224160909220977977440725226188e-01 3.31057698305513958825230247384752e-01 8.22925995577372887748723861989220e-04
            6.57224160909220977977440725226188e-01 1.17181407852650371764768877369534e-02 8.22925995577372887748723861989220e-04
            1.71165960353522994541464186113444e-01 1.19825159275385270674085802511399e-01 1.85595243546373854168218819893355e-03
            1.71165960353522994541464186113444e-01 7.09008880371091776417813434818527e-01 1.85595243546373854168218819893355e-03
            1.19825159275385270674085802511399e-01 1.71165960353522994541464186113444e-01 1.85595243546373854168218819893355e-03
            1.19825159275385270674085802511399e-01 7.09008880371091776417813434818527e-01 1.85595243546373854168218819893355e-03
            7.09008880371091776417813434818527e-01 1.71165960353522994541464186113444e-01 1.85595243546373854168218819893355e-03
            7.09008880371091776417813434818527e-01 1.19825159275385270674085802511399e-01 1.85595243546373854168218819893355e-03
            2.14838093540415636661933262985258e-01 1.18084740509011697912056249037960e-02 7.09287803930805133174886112357171e-04
            2.14838093540415636661933262985258e-01 7.73353432408683261201076675206423e-01 7.09287803930805133174886112357171e-04
            1.18084740509011697912056249037960e-02 2.14838093540415636661933262985258e-01 7.09287803930805133174886112357171e-04
            1.18084740509011697912056249037960e-02 7.73353432408683261201076675206423e-01 7.09287803930805133174886112357171e-04
            7.73353432408683261201076675206423e-01 2.14838093540415636661933262985258e-01 7.09287803930805133174886112357171e-04
            7.73353432408683261201076675206423e-01 1.18084740509011697912056249037960e-02 7.09287803930805133174886112357171e-04
            1.64032595788237256506647554488154e-01 1.18982357159540252933282999947551e-02 6.44529097785457921396257141566366e-04
            1.64032595788237256506647554488154e-01 8.24069168495808690444448529888177e-01 6.44529097785457921396257141566366e-04
            1.18982357159540252933282999947551e-02 1.64032595788237256506647554488154e-01 6.44529097785457921396257141566366e-04
            1.18982357159540252933282999947551e-02 8.24069168495808690444448529888177e-01 6.44529097785457921396257141566366e-04
            8.24069168495808690444448529888177e-01 1.64032595788237256506647554488154e-01 6.44529097785457921396257141566366e-04
            8.24069168495808690444448529888177e-01 1.18982357159540252933282999947551e-02 6.44529097785457921396257141566366e-04
            2.70599520519274439056545134008047e-01 1.18476002079960143070769262862996e-02 7.80745027452894131345106920605303e-04
            2.70599520519274439056545134008047e-01 7.17552879272729593473911791079445e-01 7.80745027452894131345106920605303e-04
            1.18476002079960143070769262862996e-02 2.70599520519274439056545134008047e-01 7.80745027452894131345106920605303e-04
            1.18476002079960143070769262862996e-02 7.17552879272729593473911791079445e-01 7.80745027452894131345106920605303e-04
            7.17552879272729593473911791079445e-01 2.70599520519274439056545134008047e-01 7.80745027452894131345106920605303e-04
            7.17552879272729593473911791079445e-01 1.18476002079960143070769262862996e-02 7.80745027452894131345106920605303e-04
            3.94881529905421246873231666540960e-01 1.15294467270000370456850546929672e-02 8.47635945473002960608155387234319e-04
            3.94881529905421246873231666540960e-01 5.93589023367578683121337235206738e-01 8.47635945473002960608155387234319e-04
            1.15294467270000370456850546929672e-02 3.94881529905421246873231666540960e-01 8.47635945473002960608155387234319e-04
            1.15294467270000370456850546929672e-02 5.93589023367578683121337235206738e-01 8.47635945473002960608155387234319e-04
            5.93589023367578683121337235206738e-01 3.94881529905421246873231666540960e-01 8.47635945473002960608155387234319e-04
            5.93589023367578683121337235206738e-01 1.15294467270000370456850546929672e-02 8.47635945473002960608155387234319e-04
            9.76801440609630089451087542329333e-02 2.24970582857548151087256904645528e-03 2.26401136283958852530098782018797e-04
            9.76801440609630089451087542329333e-02 9.00070150110461431047781388770090e-01 2.26401136283958852530098782018797e-04
            2.24970582857548151087256904645528e-03 9.76801440609630089451087542329333e-02 2.26401136283958852530098782018797e-04
            2.24970582857548151087256904645528e-03 9.00070150110461431047781388770090e-01 2.26401136283958852530098782018797e-04
            9.00070150110461431047781388770090e-01 9.76801440609630089451087542329333e-02 2.26401136283958852530098782018797e-04
            9.00070150110461431047781388770090e-01 2.24970582857548151087256904645528e-03 2.26401136283958852530098782018797e-04
            4.31663188476496095802303898381069e-01 2.17808311913864462508461095069379e-03 3.72669915533183722031185247658414e-04
            4.31663188476496095802303898381069e-01 5.66158728404365207964588080358226e-01 3.72669915533183722031185247658414e-04
            2.17808311913864462508461095069379e-03 4.31663188476496095802303898381069e-01 3.72669915533183722031185247658414e-04
            2.17808311913864462508461095069379e-03 5.66158728404365207964588080358226e-01 3.72669915533183722031185247658414e-04
            5.66158728404365207964588080358226e-01 4.31663188476496095802303898381069e-01 3.72669915533183722031185247658414e-04
            5.66158728404365207964588080358226e-01 2.17808311913864462508461095069379e-03 3.72669915533183722031185247658414e-04
            1.40323860572580283445631721406244e-01 2.25943109747260215125486126908072e-03 2.66174073629336016208468684851596e-04
            1.40323860572580283445631721406244e-01 8.57416708329947097055878657556605e-01 2.66174073629336016208468684851596e-04
            2.25943109747260215125486126908072e-03 1.40323860572580283445631721406244e-01 2.66174073629336016208468684851596e-04
            2.25943109747260215125486126908072e-03 8.57416708329947097055878657556605e-01 2.66174073629336016208468684851596e-04
            8.57416708329947097055878657556605e-01 1.40323860572580283445631721406244e-01 2.66174073629336016208468684851596e-04
            8.57416708329947097055878657556605e-01 2.25943109747260215125486126908072e-03 2.66174073629336016208468684851596e-04
            2.43796900107501685051047957131232e-01 2.24895104491058166543138163717686e-03 3.28809841813198307138715881237090e-04
            2.43796900107501685051047957131232e-01 7.53954148847587712900519818504108e-01 3.28809841813198307138715881237090e-04
            2.24895104491058166543138163717686e-03 2.43796900107501685051047957131232e-01 3.28809841813198307138715881237090e-04
            2.24895104491058166543138163717686e-03 7.53954148847587712900519818504108e-01 3.28809841813198307138715881237090e-04
            7.53954148847587712900519818504108e-01 2.43796900107501685051047957131232e-01 3.28809841813198307138715881237090e-04
            7.53954148847587712900519818504108e-01 2.24895104491058166543138163717686e-03 3.28809841813198307138715881237090e-04
            6.19939934379152152166270184352470e-02 2.21078294406797885157667238331669e-03 1.81034525565338646948404521808129e-04
            6.19939934379152152166270184352470e-02 9.35795223618016813738051951077068e-01 1.81034525565338646948404521808129e-04
            2.21078294406797885157667238331669e-03 6.19939934379152152166270184352470e-02 1.81034525565338646948404521808129e-04
            2.21078294406797885157667238331669e-03 9.35795223618016813738051951077068e-01 1.81034525565338646948404521808129e-04
            9.35795223618016813738051951077068e-01 6.19939934379152152166270184352470e-02 1.81034525565338646948404521808129e-04
            9.35795223618016813738051951077068e-01 2.21078294406797885157667238331669e-03 1.81034525565338646948404521808129e-04
            3.02950492841990437486998644089908e-01 2.24993142954393399515766738261391e-03 3.53005972835246806945663777810296e-04
            3.02950492841990437486998644089908e-01 6.94799575728465734769656592106912e-01 3.53005972835246806945663777810296e-04
            2.24993142954393399515766738261391e-03 3.02950492841990437486998644089908e-01 3.53005972835246806945663777810296e-04
            2.24993142954393399515766738261391e-03 6.94799575728465734769656592106912e-01 3.53005972835246806945663777810296e-04
            6.94799575728465734769656592106912e-01 3.02950492841990437486998644089908e-01 3.53005972835246806945663777810296e-04
            6.94799575728465734769656592106912e-01 2.24993142954393399515766738261391e-03 3.53005972835246806945663777810296e-04
            1.89266100502910289415936517798400e-01 2.26403926204082287659824501702133e-03 3.00938551109985823715786690257801e-04
            1.89266100502910289415936517798400e-01 8.08469860235048964902659918152494e-01 3.00938551109985823715786690257801e-04
            2.26403926204082287659824501702133e-03 1.89266100502910289415936517798400e-01 3.00938551109985823715786690257801e-04
            2.26403926204082287659824501702133e-03 8.08469860235048964902659918152494e-01 3.00938551109985823715786690257801e-04
            8.08469860235048964902659918152494e-01 1.89266100502910289415936517798400e-01 3.00938551109985823715786690257801e-04
            8.08469860235048964902659918152494e-01 2.26403926204082287659824501702133e-03 3.00938551109985823715786690257801e-04
            3.65919027936809082568458961759461e-01 2.21351192100762176814821025061519e-03 3.66319673274832953435709015366228e-04
            3.65919027936809082568458961759461e-01 6.31867460142183312576946718763793e-01 3.66319673274832953435709015366228e-04
            2.21351192100762176814821025061519e-03 3.65919027936809082568458961759461e-01 3.66319673274832953435709015366228e-04
            2.21351192100762176814821025061519e-03 6.31867460142183312576946718763793e-01 3.66319673274832953435709015366228e-04
            6.31867460142183312576946718763793e-01 3.65919027936809082568458961759461e-01 3.66319673274832953435709015366228e-04
            6.31867460142183312576946718763793e-01 2.21351192100762176814821025061519e-03 3.66319673274832953435709015366228e-04
            ];
        
    case  48
        
        % ALG. DEG.:   48
        % PTS CARD.:  423
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.5e-15
        
        xyw=[
            3.47724458710809425365795277684811e-01 3.47724458710809425365795277684811e-01 2.18256937689137106572978552776476e-03
            3.47724458710809425365795277684811e-01 3.04551082578381149268409444630379e-01 2.18256937689137106572978552776476e-03
            3.04551082578381149268409444630379e-01 3.47724458710809425365795277684811e-01 2.18256937689137106572978552776476e-03
            3.73431271255322050350855533906724e-01 3.73431271255322050350855533906724e-01 1.98458752265129366801343913095934e-03
            3.73431271255322050350855533906724e-01 2.53137457489355899298288932186551e-01 1.98458752265129366801343913095934e-03
            2.53137457489355899298288932186551e-01 3.73431271255322050350855533906724e-01 1.98458752265129366801343913095934e-03
            4.89339053562853232293150540499482e-01 4.89339053562853232293150540499482e-01 9.11057268392256066995293917187837e-04
            4.89339053562853232293150540499482e-01 2.13218928742935354136989190010354e-02 9.11057268392256066995293917187837e-04
            2.13218928742935354136989190010354e-02 4.89339053562853232293150540499482e-01 9.11057268392256066995293917187837e-04
            4.84109248251398810447554410529847e-02 4.84109248251398810447554410529847e-02 6.05723625032490351448311027837690e-04
            4.84109248251398810447554410529847e-02 9.03178150349720265666064733522944e-01 6.05723625032490351448311027837690e-04
            9.03178150349720265666064733522944e-01 4.84109248251398810447554410529847e-02 6.05723625032490351448311027837690e-04
            8.01595487479755675064296838172595e-02 8.01595487479755675064296838172595e-02 9.65477244813535938339210673575508e-04
            8.01595487479755675064296838172595e-02 8.39680902504048809475989401107654e-01 9.65477244813535938339210673575508e-04
            8.39680902504048809475989401107654e-01 8.01595487479755675064296838172595e-02 9.65477244813535938339210673575508e-04
            4.79090265959213501556490655275411e-01 4.79090265959213501556490655275411e-01 1.32001218782053795011699470762778e-03
            4.79090265959213501556490655275411e-01 4.18194680815729968870186894491781e-02 1.32001218782053795011699470762778e-03
            4.18194680815729968870186894491781e-02 4.79090265959213501556490655275411e-01 1.32001218782053795011699470762778e-03
            4.66642415348434647093256444350118e-01 4.66642415348434647093256444350118e-01 1.64209604320303642448208858439784e-03
            4.66642415348434647093256444350118e-01 6.67151693031307058134871112997644e-02 1.64209604320303642448208858439784e-03
            6.67151693031307058134871112997644e-02 4.66642415348434647093256444350118e-01 1.64209604320303642448208858439784e-03
            1.17045744671133034736643097062370e-02 1.17045744671133034736643097062370e-02 1.61472479217407985851648333230912e-04
            1.17045744671133034736643097062370e-02 9.76590851065773368766542716912227e-01 1.61472479217407985851648333230912e-04
            9.76590851065773368766542716912227e-01 1.17045744671133034736643097062370e-02 1.61472479217407985851648333230912e-04
            2.67846513711168798810291491463431e-01 2.67846513711168798810291491463431e-01 2.71552315199755653699065227613119e-03
            2.67846513711168798810291491463431e-01 4.64306972577662402379417017073138e-01 2.71552315199755653699065227613119e-03
            4.64306972577662402379417017073138e-01 2.67846513711168798810291491463431e-01 2.71552315199755653699065227613119e-03
            2.13138272480190765767460447932535e-01 2.13138272480190765767460447932535e-01 2.57628027224190850677509878607907e-03
            2.13138272480190765767460447932535e-01 5.73723455039618412953927872877102e-01 2.57628027224190850677509878607907e-03
            5.73723455039618412953927872877102e-01 2.13138272480190765767460447932535e-01 2.57628027224190850677509878607907e-03
            4.33626661366610932635268227386405e-01 4.33626661366610932635268227386405e-01 2.49650257497091836861313751683156e-03
            4.33626661366610932635268227386405e-01 1.32746677266778134729463545227190e-01 2.49650257497091836861313751683156e-03
            1.32746677266778134729463545227190e-01 4.33626661366610932635268227386405e-01 2.49650257497091836861313751683156e-03
            1.59900914440538832739235886037932e-01 1.59900914440538832739235886037932e-01 2.12174924592708254861284089543005e-03
            1.59900914440538832739235886037932e-01 6.80198171118922334521528227924136e-01 2.12174924592708254861284089543005e-03
            6.80198171118922334521528227924136e-01 1.59900914440538832739235886037932e-01 2.12174924592708254861284089543005e-03
            2.80636634926781647347837633788004e-02 2.80636634926781647347837633788004e-02 4.43800280512310265507158746345340e-04
            2.80636634926781647347837633788004e-02 9.43872673014643615019281241984572e-01 4.43800280512310265507158746345340e-04
            9.43872673014643615019281241984572e-01 2.80636634926781647347837633788004e-02 4.43800280512310265507158746345340e-04
            1.08047129242305917085076316652703e-01 1.08047129242305917085076316652703e-01 1.58696417448596512576874939526306e-03
            1.08047129242305917085076316652703e-01 7.83905741515388165829847366694594e-01 1.58696417448596512576874939526306e-03
            7.83905741515388165829847366694594e-01 1.08047129242305917085076316652703e-01 1.58696417448596512576874939526306e-03
            4.97175637238466083811516682544607e-01 4.97175637238466083811516682544607e-01 6.47539955694782895685712720279525e-04
            4.97175637238466083811516682544607e-01 5.64872552306783237696663491078652e-03 6.47539955694782895685712720279525e-04
            5.64872552306783237696663491078652e-03 4.97175637238466083811516682544607e-01 6.47539955694782895685712720279525e-04
            3.69567291590166213666179828578606e-01 2.15700644169616900791552893679182e-01 1.85996570614835625076532998889434e-03
            3.69567291590166213666179828578606e-01 4.14732064240216913297842893371126e-01 1.85996570614835625076532998889434e-03
            2.15700644169616900791552893679182e-01 3.69567291590166213666179828578606e-01 1.85996570614835625076532998889434e-03
            2.15700644169616900791552893679182e-01 4.14732064240216913297842893371126e-01 1.85996570614835625076532998889434e-03
            4.14732064240216913297842893371126e-01 3.69567291590166213666179828578606e-01 1.85996570614835625076532998889434e-03
            4.14732064240216913297842893371126e-01 2.15700644169616900791552893679182e-01 1.85996570614835625076532998889434e-03
            3.08481897993362230714353700022912e-01 1.08301499922003982567808577641699e-01 1.51630547228510373022891677408097e-03
            3.08481897993362230714353700022912e-01 5.83216602084633772840049914520932e-01 1.51630547228510373022891677408097e-03
            1.08301499922003982567808577641699e-01 3.08481897993362230714353700022912e-01 1.51630547228510373022891677408097e-03
            1.08301499922003982567808577641699e-01 5.83216602084633772840049914520932e-01 1.51630547228510373022891677408097e-03
            5.83216602084633772840049914520932e-01 3.08481897993362230714353700022912e-01 1.51630547228510373022891677408097e-03
            5.83216602084633772840049914520932e-01 1.08301499922003982567808577641699e-01 1.51630547228510373022891677408097e-03
            3.19642729224551336297821535481489e-01 3.46959527711364415503147995423205e-02 9.23505432416745993852436047433230e-04
            3.19642729224551336297821535481489e-01 6.45661318004312145824030721996678e-01 9.23505432416745993852436047433230e-04
            3.46959527711364415503147995423205e-02 3.19642729224551336297821535481489e-01 9.23505432416745993852436047433230e-04
            3.46959527711364415503147995423205e-02 6.45661318004312145824030721996678e-01 9.23505432416745993852436047433230e-04
            6.45661318004312145824030721996678e-01 3.19642729224551336297821535481489e-01 9.23505432416745993852436047433230e-04
            6.45661318004312145824030721996678e-01 3.46959527711364415503147995423205e-02 9.23505432416745993852436047433230e-04
            9.96255965477151894660323705465999e-01 1.34606393094949803625193229805745e-03 1.18864540424329079282544235907082e-05
            9.96255965477151894660323705465999e-01 2.39797059189860752026479673304493e-03 1.18864540424329079282544235907082e-05
            1.34606393094949803625193229805745e-03 9.96255965477151894660323705465999e-01 1.18864540424329079282544235907082e-05
            1.34606393094949803625193229805745e-03 2.39797059189860752026479673304493e-03 1.18864540424329079282544235907082e-05
            2.39797059189860752026479673304493e-03 9.96255965477151894660323705465999e-01 1.18864540424329079282544235907082e-05
            2.39797059189860752026479673304493e-03 1.34606393094949803625193229805745e-03 1.18864540424329079282544235907082e-05
            3.28084817808831308294514883527881e-01 1.91707840803216067460112981279963e-02 7.72483726320484249266984377868539e-04
            3.28084817808831308294514883527881e-01 6.52744398110847057203898202715209e-01 7.72483726320484249266984377868539e-04
            1.91707840803216067460112981279963e-02 3.28084817808831308294514883527881e-01 7.72483726320484249266984377868539e-04
            1.91707840803216067460112981279963e-02 6.52744398110847057203898202715209e-01 7.72483726320484249266984377868539e-04
            6.52744398110847057203898202715209e-01 3.28084817808831308294514883527881e-01 7.72483726320484249266984377868539e-04
            6.52744398110847057203898202715209e-01 1.91707840803216067460112981279963e-02 7.72483726320484249266984377868539e-04
            3.19812908272719464974898073705845e-01 2.76361565310809742523900922606117e-01 2.51594349134969886075396949820515e-03
            3.19812908272719464974898073705845e-01 4.03825526416470792501201003688038e-01 2.51594349134969886075396949820515e-03
            2.76361565310809742523900922606117e-01 3.19812908272719464974898073705845e-01 2.51594349134969886075396949820515e-03
            2.76361565310809742523900922606117e-01 4.03825526416470792501201003688038e-01 2.51594349134969886075396949820515e-03
            4.03825526416470792501201003688038e-01 3.19812908272719464974898073705845e-01 2.51594349134969886075396949820515e-03
            4.03825526416470792501201003688038e-01 2.76361565310809742523900922606117e-01 2.51594349134969886075396949820515e-03
            1.58926601361820019064907683059573e-01 1.64116834381746313879790744039155e-03 1.94367883400491949473509745160982e-04
            1.58926601361820019064907683059573e-01 8.39432230294362535794050472759409e-01 1.94367883400491949473509745160982e-04
            1.64116834381746313879790744039155e-03 1.58926601361820019064907683059573e-01 1.94367883400491949473509745160982e-04
            1.64116834381746313879790744039155e-03 8.39432230294362535794050472759409e-01 1.94367883400491949473509745160982e-04
            8.39432230294362535794050472759409e-01 1.58926601361820019064907683059573e-01 1.94367883400491949473509745160982e-04
            8.39432230294362535794050472759409e-01 1.64116834381746313879790744039155e-03 1.94367883400491949473509745160982e-04
            1.96275977197235607096814646865823e-01 5.33231934213235156394539160373824e-02 1.14648466765973405454004385717326e-03
            1.96275977197235607096814646865823e-01 7.50400829381440925835988764447393e-01 1.14648466765973405454004385717326e-03
            5.33231934213235156394539160373824e-02 1.96275977197235607096814646865823e-01 1.14648466765973405454004385717326e-03
            5.33231934213235156394539160373824e-02 7.50400829381440925835988764447393e-01 1.14648466765973405454004385717326e-03
            7.50400829381440925835988764447393e-01 1.96275977197235607096814646865823e-01 1.14648466765973405454004385717326e-03
            7.50400829381440925835988764447393e-01 5.33231934213235156394539160373824e-02 1.14648466765973405454004385717326e-03
            2.93342184590181742631642691776506e-01 8.29288364011383782870012737475918e-02 1.62950409788547570175132328529344e-03
            2.93342184590181742631642691776506e-01 6.23728979008679851325780418846989e-01 1.62950409788547570175132328529344e-03
            8.29288364011383782870012737475918e-02 2.93342184590181742631642691776506e-01 1.62950409788547570175132328529344e-03
            8.29288364011383782870012737475918e-02 6.23728979008679851325780418846989e-01 1.62950409788547570175132328529344e-03
            6.23728979008679851325780418846989e-01 2.93342184590181742631642691776506e-01 1.62950409788547570175132328529344e-03
            6.23728979008679851325780418846989e-01 8.29288364011383782870012737475918e-02 1.62950409788547570175132328529344e-03
            1.24727325881397566043062852259027e-01 2.33190171491815048665507958958187e-02 6.49717196700759806390901296424545e-04
            1.24727325881397566043062852259027e-01 8.51953656969420936029280255752383e-01 6.49717196700759806390901296424545e-04
            2.33190171491815048665507958958187e-02 1.24727325881397566043062852259027e-01 6.49717196700759806390901296424545e-04
            2.33190171491815048665507958958187e-02 8.51953656969420936029280255752383e-01 6.49717196700759806390901296424545e-04
            8.51953656969420936029280255752383e-01 1.24727325881397566043062852259027e-01 6.49717196700759806390901296424545e-04
            8.51953656969420936029280255752383e-01 2.33190171491815048665507958958187e-02 6.49717196700759806390901296424545e-04
            3.11593245943279584242446844655205e-01 1.38713093959385580511423086136347e-01 2.02211589957075653714579921427230e-03
            3.11593245943279584242446844655205e-01 5.49693660097334890757281300466275e-01 2.02211589957075653714579921427230e-03
            1.38713093959385580511423086136347e-01 3.11593245943279584242446844655205e-01 2.02211589957075653714579921427230e-03
            1.38713093959385580511423086136347e-01 5.49693660097334890757281300466275e-01 2.02211589957075653714579921427230e-03
            5.49693660097334890757281300466275e-01 3.11593245943279584242446844655205e-01 2.02211589957075653714579921427230e-03
            5.49693660097334890757281300466275e-01 1.38713093959385580511423086136347e-01 2.02211589957075653714579921427230e-03
            1.10541441031880752210447838024265e-01 1.05406394081225263104073519571102e-02 4.32239553547260173249622283719873e-04
            1.10541441031880752210447838024265e-01 8.78917919559996652090205770946341e-01 4.32239553547260173249622283719873e-04
            1.05406394081225263104073519571102e-02 1.10541441031880752210447838024265e-01 4.32239553547260173249622283719873e-04
            1.05406394081225263104073519571102e-02 8.78917919559996652090205770946341e-01 4.32239553547260173249622283719873e-04
            8.78917919559996652090205770946341e-01 1.10541441031880752210447838024265e-01 4.32239553547260173249622283719873e-04
            8.78917919559996652090205770946341e-01 1.05406394081225263104073519571102e-02 4.32239553547260173249622283719873e-04
            2.97417196275917228298624195304001e-01 1.03472505675079231440838256617099e-02 6.32985718237871709465625080781592e-04
            2.97417196275917228298624195304001e-01 6.92235553156574812128098983521340e-01 6.32985718237871709465625080781592e-04
            1.03472505675079231440838256617099e-02 2.97417196275917228298624195304001e-01 6.32985718237871709465625080781592e-04
            1.03472505675079231440838256617099e-02 6.92235553156574812128098983521340e-01 6.32985718237871709465625080781592e-04
            6.92235553156574812128098983521340e-01 2.97417196275917228298624195304001e-01 6.32985718237871709465625080781592e-04
            6.92235553156574812128098983521340e-01 1.03472505675079231440838256617099e-02 6.32985718237871709465625080781592e-04
            2.94068058652278530029633429876412e-01 5.37818816587767034587130865475046e-02 1.37129429762075758132688907409147e-03
            2.94068058652278530029633429876412e-01 6.52150059688944683244926636689343e-01 1.37129429762075758132688907409147e-03
            5.37818816587767034587130865475046e-02 2.94068058652278530029633429876412e-01 1.37129429762075758132688907409147e-03
            5.37818816587767034587130865475046e-02 6.52150059688944683244926636689343e-01 1.37129429762075758132688907409147e-03
            6.52150059688944683244926636689343e-01 2.94068058652278530029633429876412e-01 1.37129429762075758132688907409147e-03
            6.52150059688944683244926636689343e-01 5.37818816587767034587130865475046e-02 1.37129429762075758132688907409147e-03
            1.11401217606310831698124275135342e-01 6.97603580204464307490397345645761e-02 1.05770377683287353405561237451593e-03
            1.11401217606310831698124275135342e-01 8.18838424373242723675048182485625e-01 1.05770377683287353405561237451593e-03
            6.97603580204464307490397345645761e-02 1.11401217606310831698124275135342e-01 1.05770377683287353405561237451593e-03
            6.97603580204464307490397345645761e-02 8.18838424373242723675048182485625e-01 1.05770377683287353405561237451593e-03
            8.18838424373242723675048182485625e-01 1.11401217606310831698124275135342e-01 1.05770377683287353405561237451593e-03
            8.18838424373242723675048182485625e-01 6.97603580204464307490397345645761e-02 1.05770377683287353405561237451593e-03
            4.43743453863629289113390541388071e-01 1.55023027792510516287283195424607e-02 8.11577263642843300429918773630789e-04
            4.43743453863629289113390541388071e-01 5.40754243357119634971752475394169e-01 8.11577263642843300429918773630789e-04
            1.55023027792510516287283195424607e-02 4.43743453863629289113390541388071e-01 8.11577263642843300429918773630789e-04
            1.55023027792510516287283195424607e-02 5.40754243357119634971752475394169e-01 8.11577263642843300429918773630789e-04
            5.40754243357119634971752475394169e-01 4.43743453863629289113390541388071e-01 8.11577263642843300429918773630789e-04
            5.40754243357119634971752475394169e-01 1.55023027792510516287283195424607e-02 8.11577263642843300429918773630789e-04
            3.67635711929402475561801111325622e-01 3.96225072588169413556613562832354e-02 1.27063686670264856977408118865469e-03
            3.67635711929402475561801111325622e-01 5.92741780811780527571386301133316e-01 1.27063686670264856977408118865469e-03
            3.96225072588169413556613562832354e-02 3.67635711929402475561801111325622e-01 1.27063686670264856977408118865469e-03
            3.96225072588169413556613562832354e-02 5.92741780811780527571386301133316e-01 1.27063686670264856977408118865469e-03
            5.92741780811780527571386301133316e-01 3.67635711929402475561801111325622e-01 1.27063686670264856977408118865469e-03
            5.92741780811780527571386301133316e-01 3.96225072588169413556613562832354e-02 1.27063686670264856977408118865469e-03
            4.83834188936242623468864110236609e-02 1.18107009236640553284658494703763e-02 3.11317736056086768765133188097138e-04
            4.83834188936242623468864110236609e-02 9.39805880182711605996814796526451e-01 3.11317736056086768765133188097138e-04
            1.18107009236640553284658494703763e-02 4.83834188936242623468864110236609e-02 3.11317736056086768765133188097138e-04
            1.18107009236640553284658494703763e-02 9.39805880182711605996814796526451e-01 3.11317736056086768765133188097138e-04
            9.39805880182711605996814796526451e-01 4.83834188936242623468864110236609e-02 3.11317736056086768765133188097138e-04
            9.39805880182711605996814796526451e-01 1.18107009236640553284658494703763e-02 3.11317736056086768765133188097138e-04
            4.26198546034848302355158011778258e-01 3.52878923463722377062978807771287e-02 1.32092018998813736860298284625514e-03
            4.26198546034848302355158011778258e-01 5.38513561618779501571907530887984e-01 1.32092018998813736860298284625514e-03
            3.52878923463722377062978807771287e-02 4.26198546034848302355158011778258e-01 1.32092018998813736860298284625514e-03
            3.52878923463722377062978807771287e-02 5.38513561618779501571907530887984e-01 1.32092018998813736860298284625514e-03
            5.38513561618779501571907530887984e-01 4.26198546034848302355158011778258e-01 1.32092018998813736860298284625514e-03
            5.38513561618779501571907530887984e-01 3.52878923463722377062978807771287e-02 1.32092018998813736860298284625514e-03
            3.46403707508342595389194684685208e-01 6.47150930019751435828823105111951e-02 1.63265855137869567853536878487830e-03
            3.46403707508342595389194684685208e-01 5.88881199489682316539074236061424e-01 1.63265855137869567853536878487830e-03
            6.47150930019751435828823105111951e-02 3.46403707508342595389194684685208e-01 1.63265855137869567853536878487830e-03
            6.47150930019751435828823105111951e-02 5.88881199489682316539074236061424e-01 1.63265855137869567853536878487830e-03
            5.88881199489682316539074236061424e-01 3.46403707508342595389194684685208e-01 1.63265855137869567853536878487830e-03
            5.88881199489682316539074236061424e-01 6.47150930019751435828823105111951e-02 1.63265855137869567853536878487830e-03
            7.61686440792014585188240971547202e-02 1.04225715676421054611822114566166e-02 3.82757292961209444771136789498200e-04
            7.61686440792014585188240971547202e-02 9.13408784353156422142205883574206e-01 3.82757292961209444771136789498200e-04
            1.04225715676421054611822114566166e-02 7.61686440792014585188240971547202e-02 3.82757292961209444771136789498200e-04
            1.04225715676421054611822114566166e-02 9.13408784353156422142205883574206e-01 3.82757292961209444771136789498200e-04
            9.13408784353156422142205883574206e-01 7.61686440792014585188240971547202e-02 3.82757292961209444771136789498200e-04
            9.13408784353156422142205883574206e-01 1.04225715676421054611822114566166e-02 3.82757292961209444771136789498200e-04
            3.19855805989206987138828708339133e-01 2.27104917530668970471197098959237e-01 2.58206804723576507573645777426918e-03
            3.19855805989206987138828708339133e-01 4.53039276480124097901125423959456e-01 2.58206804723576507573645777426918e-03
            2.27104917530668970471197098959237e-01 3.19855805989206987138828708339133e-01 2.58206804723576507573645777426918e-03
            2.27104917530668970471197098959237e-01 4.53039276480124097901125423959456e-01 2.58206804723576507573645777426918e-03
            4.53039276480124097901125423959456e-01 3.19855805989206987138828708339133e-01 2.58206804723576507573645777426918e-03
            4.53039276480124097901125423959456e-01 2.27104917530668970471197098959237e-01 2.58206804723576507573645777426918e-03
            1.14976749976327827229560796240548e-01 4.31548880805705872831801173106214e-02 9.27719490637918990996735324472411e-04
            1.14976749976327827229560796240548e-01 8.41868361943101661815092029428342e-01 9.27719490637918990996735324472411e-04
            4.31548880805705872831801173106214e-02 1.14976749976327827229560796240548e-01 9.27719490637918990996735324472411e-04
            4.31548880805705872831801173106214e-02 8.41868361943101661815092029428342e-01 9.27719490637918990996735324472411e-04
            8.41868361943101661815092029428342e-01 1.14976749976327827229560796240548e-01 9.27719490637918990996735324472411e-04
            8.41868361943101661815092029428342e-01 4.31548880805705872831801173106214e-02 9.27719490637918990996735324472411e-04
            1.55341187486334963985257218155311e-01 4.89749482183904280563346844701300e-02 1.13990500568115723546702700019750e-03
            1.55341187486334963985257218155311e-01 7.95683864295274601019514193467330e-01 1.13990500568115723546702700019750e-03
            4.89749482183904280563346844701300e-02 1.55341187486334963985257218155311e-01 1.13990500568115723546702700019750e-03
            4.89749482183904280563346844701300e-02 7.95683864295274601019514193467330e-01 1.13990500568115723546702700019750e-03
            7.95683864295274601019514193467330e-01 1.55341187486334963985257218155311e-01 1.13990500568115723546702700019750e-03
            7.95683864295274601019514193467330e-01 4.89749482183904280563346844701300e-02 1.13990500568115723546702700019750e-03
            1.53664139478065231436687554378295e-01 9.45454884202667593384727950933666e-03 5.13397308598045627042738558998280e-04
            1.53664139478065231436687554378295e-01 8.36881311679908090894741690135561e-01 5.13397308598045627042738558998280e-04
            9.45454884202667593384727950933666e-03 1.53664139478065231436687554378295e-01 5.13397308598045627042738558998280e-04
            9.45454884202667593384727950933666e-03 8.36881311679908090894741690135561e-01 5.13397308598045627042738558998280e-04
            8.36881311679908090894741690135561e-01 1.53664139478065231436687554378295e-01 5.13397308598045627042738558998280e-04
            8.36881311679908090894741690135561e-01 9.45454884202667593384727950933666e-03 5.13397308598045627042738558998280e-04
            7.98538207718234027687742582202191e-02 1.89430587911264334925742769399903e-03 1.66421560563311383930049092150227e-04
            7.98538207718234027687742582202191e-02 9.18251873349063996165853041020455e-01 1.66421560563311383930049092150227e-04
            1.89430587911264334925742769399903e-03 7.98538207718234027687742582202191e-02 1.66421560563311383930049092150227e-04
            1.89430587911264334925742769399903e-03 9.18251873349063996165853041020455e-01 1.66421560563311383930049092150227e-04
            9.18251873349063996165853041020455e-01 7.98538207718234027687742582202191e-02 1.66421560563311383930049092150227e-04
            9.18251873349063996165853041020455e-01 1.89430587911264334925742769399903e-03 1.66421560563311383930049092150227e-04
            1.97860112128548371046576903609093e-01 1.20283224036929169142862150465589e-02 6.07874655725859571082925203455716e-04
            1.97860112128548371046576903609093e-01 7.90111565467758736325265545019647e-01 6.07874655725859571082925203455716e-04
            1.20283224036929169142862150465589e-02 1.97860112128548371046576903609093e-01 6.07874655725859571082925203455716e-04
            1.20283224036929169142862150465589e-02 7.90111565467758736325265545019647e-01 6.07874655725859571082925203455716e-04
            7.90111565467758736325265545019647e-01 1.97860112128548371046576903609093e-01 6.07874655725859571082925203455716e-04
            7.90111565467758736325265545019647e-01 1.20283224036929169142862150465589e-02 6.07874655725859571082925203455716e-04
            3.16915380380781819269486732082441e-01 1.90446777793631645894767423499161e-03 2.94655786447839092897366120027414e-04
            3.16915380380781819269486732082441e-01 6.81180151841281888991375126352068e-01 2.94655786447839092897366120027414e-04
            1.90446777793631645894767423499161e-03 3.16915380380781819269486732082441e-01 2.94655786447839092897366120027414e-04
            1.90446777793631645894767423499161e-03 6.81180151841281888991375126352068e-01 2.94655786447839092897366120027414e-04
            6.81180151841281888991375126352068e-01 3.16915380380781819269486732082441e-01 2.94655786447839092897366120027414e-04
            6.81180151841281888991375126352068e-01 1.90446777793631645894767423499161e-03 2.94655786447839092897366120027414e-04
            1.64563977163413477367015502750291e-01 2.62052237923564648736096671655105e-02 8.67132902247431448274073151338825e-04
            1.64563977163413477367015502750291e-01 8.09230799044230009187117502733599e-01 8.67132902247431448274073151338825e-04
            2.62052237923564648736096671655105e-02 1.64563977163413477367015502750291e-01 8.67132902247431448274073151338825e-04
            2.62052237923564648736096671655105e-02 8.09230799044230009187117502733599e-01 8.67132902247431448274073151338825e-04
            8.09230799044230009187117502733599e-01 1.64563977163413477367015502750291e-01 8.67132902247431448274073151338825e-04
            8.09230799044230009187117502733599e-01 2.62052237923564648736096671655105e-02 8.67132902247431448274073151338825e-04
            3.61129835979937918732929347243044e-01 9.44709215240264049695539938511502e-02 2.02493753788023263251449890276490e-03
            3.61129835979937918732929347243044e-01 5.44399242496035662419728851091349e-01 2.02493753788023263251449890276490e-03
            9.44709215240264049695539938511502e-02 3.61129835979937918732929347243044e-01 2.02493753788023263251449890276490e-03
            9.44709215240264049695539938511502e-02 5.44399242496035662419728851091349e-01 2.02493753788023263251449890276490e-03
            5.44399242496035662419728851091349e-01 3.61129835979937918732929347243044e-01 2.02493753788023263251449890276490e-03
            5.44399242496035662419728851091349e-01 9.44709215240264049695539938511502e-02 2.02493753788023263251449890276490e-03
            4.19825977454229393170948014812893e-01 9.63233490337837422456956915084447e-02 2.13711614885642468483761824415978e-03
            4.19825977454229393170948014812893e-01 4.83850673511986795194417254606378e-01 2.13711614885642468483761824415978e-03
            9.63233490337837422456956915084447e-02 4.19825977454229393170948014812893e-01 2.13711614885642468483761824415978e-03
            9.63233490337837422456956915084447e-02 4.83850673511986795194417254606378e-01 2.13711614885642468483761824415978e-03
            4.83850673511986795194417254606378e-01 4.19825977454229393170948014812893e-01 2.13711614885642468483761824415978e-03
            4.83850673511986795194417254606378e-01 9.63233490337837422456956915084447e-02 2.13711614885642468483761824415978e-03
            2.43175224574663090271187115831708e-01 5.34903172634274073904414592561807e-02 1.41675427379012392262169761636414e-03
            2.43175224574663090271187115831708e-01 7.03334458161909426010538481932599e-01 1.41675427379012392262169761636414e-03
            5.34903172634274073904414592561807e-02 2.43175224574663090271187115831708e-01 1.41675427379012392262169761636414e-03
            5.34903172634274073904414592561807e-02 7.03334458161909426010538481932599e-01 1.41675427379012392262169761636414e-03
            7.03334458161909426010538481932599e-01 2.43175224574663090271187115831708e-01 1.41675427379012392262169761636414e-03
            7.03334458161909426010538481932599e-01 5.34903172634274073904414592561807e-02 1.41675427379012392262169761636414e-03
            1.45983223113643451052112709476205e-01 7.95244786008371812302542025463481e-02 1.37190627294187265580727164149266e-03
            1.45983223113643451052112709476205e-01 7.74492298285519353839845280162990e-01 1.37190627294187265580727164149266e-03
            7.95244786008371812302542025463481e-02 1.45983223113643451052112709476205e-01 1.37190627294187265580727164149266e-03
            7.95244786008371812302542025463481e-02 7.74492298285519353839845280162990e-01 1.37190627294187265580727164149266e-03
            7.74492298285519353839845280162990e-01 1.45983223113643451052112709476205e-01 1.37190627294187265580727164149266e-03
            7.74492298285519353839845280162990e-01 7.95244786008371812302542025463481e-02 1.37190627294187265580727164149266e-03
            2.72392832679802729400897476352839e-02 1.10307547814719290879548196926407e-02 2.36510728177526303485367353829361e-04
            2.72392832679802729400897476352839e-02 9.61729961950547718174675537738949e-01 2.36510728177526303485367353829361e-04
            1.10307547814719290879548196926407e-02 2.72392832679802729400897476352839e-02 2.36510728177526303485367353829361e-04
            1.10307547814719290879548196926407e-02 9.61729961950547718174675537738949e-01 2.36510728177526303485367353829361e-04
            9.61729961950547718174675537738949e-01 2.72392832679802729400897476352839e-02 2.36510728177526303485367353829361e-04
            9.61729961950547718174675537738949e-01 1.10307547814719290879548196926407e-02 2.36510728177526303485367353829361e-04
            2.45313708964778415255736376821005e-01 1.21862329639788515706078086964226e-02 6.86658872080503661014527772010752e-04
            2.45313708964778415255736376821005e-01 7.42500058071242752255614050227450e-01 6.86658872080503661014527772010752e-04
            1.21862329639788515706078086964226e-02 2.45313708964778415255736376821005e-01 6.86658872080503661014527772010752e-04
            1.21862329639788515706078086964226e-02 7.42500058071242752255614050227450e-01 6.86658872080503661014527772010752e-04
            7.42500058071242752255614050227450e-01 2.45313708964778415255736376821005e-01 6.86658872080503661014527772010752e-04
            7.42500058071242752255614050227450e-01 1.21862329639788515706078086964226e-02 6.86658872080503661014527772010752e-04
            1.16554219073452897115217297141498e-01 2.11321826247760956654642150454038e-03 2.11650495730307140404558019497472e-04
            1.16554219073452897115217297141498e-01 8.81332562664069563140856189420447e-01 2.11650495730307140404558019497472e-04
            2.11321826247760956654642150454038e-03 1.16554219073452897115217297141498e-01 2.11650495730307140404558019497472e-04
            2.11321826247760956654642150454038e-03 8.81332562664069563140856189420447e-01 2.11650495730307140404558019497472e-04
            8.81332562664069563140856189420447e-01 1.16554219073452897115217297141498e-01 2.11650495730307140404558019497472e-04
            8.81332562664069563140856189420447e-01 2.11321826247760956654642150454038e-03 2.11650495730307140404558019497472e-04
            5.35856828700446083391106810722704e-02 2.72189906580712175088176962844955e-02 5.33230593581922643862947008841502e-04
            5.35856828700446083391106810722704e-02 9.19195326471884177621518574596848e-01 5.33230593581922643862947008841502e-04
            2.72189906580712175088176962844955e-02 5.35856828700446083391106810722704e-02 5.33230593581922643862947008841502e-04
            2.72189906580712175088176962844955e-02 9.19195326471884177621518574596848e-01 5.33230593581922643862947008841502e-04
            9.19195326471884177621518574596848e-01 5.35856828700446083391106810722704e-02 5.33230593581922643862947008841502e-04
            9.19195326471884177621518574596848e-01 2.72189906580712175088176962844955e-02 5.33230593581922643862947008841502e-04
            8.51355181500168534114436624804512e-02 2.62562590124720725281992628197258e-02 6.91200536655852348054407396915622e-04
            8.51355181500168534114436624804512e-02 8.88608222837511063652016218838980e-01 6.91200536655852348054407396915622e-04
            2.62562590124720725281992628197258e-02 8.51355181500168534114436624804512e-02 6.91200536655852348054407396915622e-04
            2.62562590124720725281992628197258e-02 8.88608222837511063652016218838980e-01 6.91200536655852348054407396915622e-04
            8.88608222837511063652016218838980e-01 8.51355181500168534114436624804512e-02 6.91200536655852348054407396915622e-04
            8.88608222837511063652016218838980e-01 2.62562590124720725281992628197258e-02 6.91200536655852348054407396915622e-04
            4.08492625785633056700874021771597e-01 6.24750974503704217810629018003965e-02 1.82925589651349810443869969844854e-03
            4.08492625785633056700874021771597e-01 5.29032276763996556212532595964149e-01 1.82925589651349810443869969844854e-03
            6.24750974503704217810629018003965e-02 4.08492625785633056700874021771597e-01 1.82925589651349810443869969844854e-03
            6.24750974503704217810629018003965e-02 5.29032276763996556212532595964149e-01 1.82925589651349810443869969844854e-03
            5.29032276763996556212532595964149e-01 4.08492625785633056700874021771597e-01 1.82925589651349810443869969844854e-03
            5.29032276763996556212532595964149e-01 6.24750974503704217810629018003965e-02 1.82925589651349810443869969844854e-03
            3.82251045094813779456899283104576e-01 1.73776209876193493686358237937384e-01 2.60500242743329267838814367053146e-03
            3.82251045094813779456899283104576e-01 4.43972745028992754612318094586954e-01 2.60500242743329267838814367053146e-03
            1.73776209876193493686358237937384e-01 3.82251045094813779456899283104576e-01 2.60500242743329267838814367053146e-03
            1.73776209876193493686358237937384e-01 4.43972745028992754612318094586954e-01 2.60500242743329267838814367053146e-03
            4.43972745028992754612318094586954e-01 3.82251045094813779456899283104576e-01 2.60500242743329267838814367053146e-03
            4.43972745028992754612318094586954e-01 1.73776209876193493686358237937384e-01 2.60500242743329267838814367053146e-03
            2.59740322065890247293928894123383e-02 2.04912604306957185293369683165565e-03 1.03655280295382562732835196595005e-04
            2.59740322065890247293928894123383e-02 9.71976841750341380432587357063312e-01 1.03655280295382562732835196595005e-04
            2.04912604306957185293369683165565e-03 2.59740322065890247293928894123383e-02 1.03655280295382562732835196595005e-04
            2.04912604306957185293369683165565e-03 9.71976841750341380432587357063312e-01 1.03655280295382562732835196595005e-04
            9.71976841750341380432587357063312e-01 2.59740322065890247293928894123383e-02 1.03655280295382562732835196595005e-04
            9.71976841750341380432587357063312e-01 2.04912604306957185293369683165565e-03 1.03655280295382562732835196595005e-04
            3.85689188760064305405705908924574e-01 1.88900416615420362054234004745012e-02 9.79158143431579485599369760961963e-04
            3.85689188760064305405705908924574e-01 5.95420769578393627163848123018397e-01 9.79158143431579485599369760961963e-04
            1.88900416615420362054234004745012e-02 3.85689188760064305405705908924574e-01 9.79158143431579485599369760961963e-04
            1.88900416615420362054234004745012e-02 5.95420769578393627163848123018397e-01 9.79158143431579485599369760961963e-04
            5.95420769578393627163848123018397e-01 3.85689188760064305405705908924574e-01 9.79158143431579485599369760961963e-04
            5.95420769578393627163848123018397e-01 1.88900416615420362054234004745012e-02 9.79158143431579485599369760961963e-04
            1.90746479431403104243258894712199e-01 8.26189529456984456112778048009204e-02 1.60446126495938120974649176986304e-03
            1.90746479431403104243258894712199e-01 7.26634567622898464023251108301338e-01 1.60446126495938120974649176986304e-03
            8.26189529456984456112778048009204e-02 1.90746479431403104243258894712199e-01 1.60446126495938120974649176986304e-03
            8.26189529456984456112778048009204e-02 7.26634567622898464023251108301338e-01 1.60446126495938120974649176986304e-03
            7.26634567622898464023251108301338e-01 1.90746479431403104243258894712199e-01 1.60446126495938120974649176986304e-03
            7.26634567622898464023251108301338e-01 8.26189529456984456112778048009204e-02 1.60446126495938120974649176986304e-03
            4.93146516048518085417740053344460e-02 2.27995678675310979241497300051833e-03 1.57207577306592641286364653296914e-04
            4.93146516048518085417740053344460e-02 9.48405391608395098579364912438905e-01 1.57207577306592641286364653296914e-04
            2.27995678675310979241497300051833e-03 4.93146516048518085417740053344460e-02 1.57207577306592641286364653296914e-04
            2.27995678675310979241497300051833e-03 9.48405391608395098579364912438905e-01 1.57207577306592641286364653296914e-04
            9.48405391608395098579364912438905e-01 4.93146516048518085417740053344460e-02 1.57207577306592641286364653296914e-04
            9.48405391608395098579364912438905e-01 2.27995678675310979241497300051833e-03 1.57207577306592641286364653296914e-04
            4.26482545498046594634189432326821e-01 4.77225212030264397955470201395656e-03 5.39257277560205414319860484084757e-04
            4.26482545498046594634189432326821e-01 5.68745202381650827305747952777892e-01 5.39257277560205414319860484084757e-04
            4.77225212030264397955470201395656e-03 4.26482545498046594634189432326821e-01 5.39257277560205414319860484084757e-04
            4.77225212030264397955470201395656e-03 5.68745202381650827305747952777892e-01 5.39257277560205414319860484084757e-04
            5.68745202381650827305747952777892e-01 4.26482545498046594634189432326821e-01 5.39257277560205414319860484084757e-04
            5.68745202381650827305747952777892e-01 4.77225212030264397955470201395656e-03 5.39257277560205414319860484084757e-04
            1.51364752670565094749832724119187e-01 1.16202356411588061946282834924205e-01 1.80021348830760805070916319436947e-03
            1.51364752670565094749832724119187e-01 7.32432890917846912692823480028892e-01 1.80021348830760805070916319436947e-03
            1.16202356411588061946282834924205e-01 1.51364752670565094749832724119187e-01 1.80021348830760805070916319436947e-03
            1.16202356411588061946282834924205e-01 7.32432890917846912692823480028892e-01 1.80021348830760805070916319436947e-03
            7.32432890917846912692823480028892e-01 1.51364752670565094749832724119187e-01 1.80021348830760805070916319436947e-03
            7.32432890917846912692823480028892e-01 1.16202356411588061946282834924205e-01 1.80021348830760805070916319436947e-03
            2.41085560390986441969829456866137e-01 8.55252642423670605964503010909539e-02 1.83059570892130899118221964982922e-03
            2.41085560390986441969829456866137e-01 6.73389175366646441922569010785082e-01 1.83059570892130899118221964982922e-03
            8.55252642423670605964503010909539e-02 2.41085560390986441969829456866137e-01 1.83059570892130899118221964982922e-03
            8.55252642423670605964503010909539e-02 6.73389175366646441922569010785082e-01 1.83059570892130899118221964982922e-03
            6.73389175366646441922569010785082e-01 2.41085560390986441969829456866137e-01 1.83059570892130899118221964982922e-03
            6.73389175366646441922569010785082e-01 8.55252642423670605964503010909539e-02 1.83059570892130899118221964982922e-03
            2.12036211205479691033204403538548e-01 2.97767883228066514156306965332988e-02 1.08085615693754135205328914537404e-03
            2.12036211205479691033204403538548e-01 7.58187000471713723470656987046823e-01 1.08085615693754135205328914537404e-03
            2.97767883228066514156306965332988e-02 2.12036211205479691033204403538548e-01 1.08085615693754135205328914537404e-03
            2.97767883228066514156306965332988e-02 7.58187000471713723470656987046823e-01 1.08085615693754135205328914537404e-03
            7.58187000471713723470656987046823e-01 2.12036211205479691033204403538548e-01 1.08085615693754135205328914537404e-03
            7.58187000471713723470656987046823e-01 2.97767883228066514156306965332988e-02 1.08085615693754135205328914537404e-03
            7.73420280842538471199887339935231e-02 5.02535370358713970317943164900498e-02 9.00528138193089941303248746606869e-04
            7.73420280842538471199887339935231e-02 8.72404434879874735031535237794742e-01 9.00528138193089941303248746606869e-04
            5.02535370358713970317943164900498e-02 7.73420280842538471199887339935231e-02 9.00528138193089941303248746606869e-04
            5.02535370358713970317943164900498e-02 8.72404434879874735031535237794742e-01 9.00528138193089941303248746606869e-04
            8.72404434879874735031535237794742e-01 7.73420280842538471199887339935231e-02 9.00528138193089941303248746606869e-04
            8.72404434879874735031535237794742e-01 5.02535370358713970317943164900498e-02 9.00528138193089941303248746606869e-04
            2.65927023654531302643277967945323e-01 2.16300207102354397070698155403079e-01 2.72377964713794214896003254011703e-03
            2.65927023654531302643277967945323e-01 5.17772769243114328041599492280511e-01 2.72377964713794214896003254011703e-03
            2.16300207102354397070698155403079e-01 2.65927023654531302643277967945323e-01 2.72377964713794214896003254011703e-03
            2.16300207102354397070698155403079e-01 5.17772769243114328041599492280511e-01 2.72377964713794214896003254011703e-03
            5.17772769243114328041599492280511e-01 2.65927023654531302643277967945323e-01 2.72377964713794214896003254011703e-03
            5.17772769243114328041599492280511e-01 2.16300207102354397070698155403079e-01 2.72377964713794214896003254011703e-03
            1.02781124040352035697853949614000e-02 2.31820768447709487508623382723272e-03 6.98926007621862555550806828108534e-05
            1.02781124040352035697853949614000e-02 9.87403679911487697218319681269350e-01 6.98926007621862555550806828108534e-05
            2.31820768447709487508623382723272e-03 1.02781124040352035697853949614000e-02 6.98926007621862555550806828108534e-05
            2.31820768447709487508623382723272e-03 9.87403679911487697218319681269350e-01 6.98926007621862555550806828108534e-05
            9.87403679911487697218319681269350e-01 1.02781124040352035697853949614000e-02 6.98926007621862555550806828108534e-05
            9.87403679911487697218319681269350e-01 2.31820768447709487508623382723272e-03 6.98926007621862555550806828108534e-05
            3.63748680489815068295200717329863e-01 6.43018778976174251726938280171453e-03 5.98100592597651135526315435697597e-04
            3.63748680489815068295200717329863e-01 6.29821131720423088573568293213611e-01 5.98100592597651135526315435697597e-04
            6.43018778976174251726938280171453e-03 3.63748680489815068295200717329863e-01 5.98100592597651135526315435697597e-04
            6.43018778976174251726938280171453e-03 6.29821131720423088573568293213611e-01 5.98100592597651135526315435697597e-04
            6.29821131720423088573568293213611e-01 3.63748680489815068295200717329863e-01 5.98100592597651135526315435697597e-04
            6.29821131720423088573568293213611e-01 6.43018778976174251726938280171453e-03 5.98100592597651135526315435697597e-04
            3.69820205106713650877736654365435e-01 1.33111115065471463658752782066585e-01 2.47628291179179289838163668946436e-03
            3.69820205106713650877736654365435e-01 4.97068679827814885463510563567979e-01 2.47628291179179289838163668946436e-03
            1.33111115065471463658752782066585e-01 3.69820205106713650877736654365435e-01 2.47628291179179289838163668946436e-03
            1.33111115065471463658752782066585e-01 4.97068679827814885463510563567979e-01 2.47628291179179289838163668946436e-03
            4.97068679827814885463510563567979e-01 3.69820205106713650877736654365435e-01 2.47628291179179289838163668946436e-03
            4.97068679827814885463510563567979e-01 1.33111115065471463658752782066585e-01 2.47628291179179289838163668946436e-03
            2.62236855625541442904591349360999e-01 1.68252485371562954252766530771623e-01 2.52760293694821334617439667624694e-03
            2.62236855625541442904591349360999e-01 5.69510659002895547331490888609551e-01 2.52760293694821334617439667624694e-03
            1.68252485371562954252766530771623e-01 2.62236855625541442904591349360999e-01 2.52760293694821334617439667624694e-03
            1.68252485371562954252766530771623e-01 5.69510659002895547331490888609551e-01 2.52760293694821334617439667624694e-03
            5.69510659002895547331490888609551e-01 2.62236855625541442904591349360999e-01 2.52760293694821334617439667624694e-03
            5.69510659002895547331490888609551e-01 1.68252485371562954252766530771623e-01 2.52760293694821334617439667624694e-03
            2.05865274198998832178020279570774e-01 2.37764370084020187565143622521191e-03 3.02795530949603256942043172372792e-04
            2.05865274198998832178020279570774e-01 7.91757082100161047044650786119746e-01 3.02795530949603256942043172372792e-04
            2.37764370084020187565143622521191e-03 2.05865274198998832178020279570774e-01 3.02795530949603256942043172372792e-04
            2.37764370084020187565143622521191e-03 7.91757082100161047044650786119746e-01 3.02795530949603256942043172372792e-04
            7.91757082100161047044650786119746e-01 2.05865274198998832178020279570774e-01 3.02795530949603256942043172372792e-04
            7.91757082100161047044650786119746e-01 2.37764370084020187565143622521191e-03 3.02795530949603256942043172372792e-04
            3.21061299239171094033196141026565e-01 1.79275176914734907240855932286649e-01 2.67889510546186161363269029322964e-03
            3.21061299239171094033196141026565e-01 4.99663523846094026481523542315699e-01 2.67889510546186161363269029322964e-03
            1.79275176914734907240855932286649e-01 3.21061299239171094033196141026565e-01 2.67889510546186161363269029322964e-03
            1.79275176914734907240855932286649e-01 4.99663523846094026481523542315699e-01 2.67889510546186161363269029322964e-03
            4.99663523846094026481523542315699e-01 3.21061299239171094033196141026565e-01 2.67889510546186161363269029322964e-03
            4.99663523846094026481523542315699e-01 1.79275176914734907240855932286649e-01 2.67889510546186161363269029322964e-03
            2.67008925352811121722851339654881e-01 2.84633516120191137899197997285228e-02 1.18276761225013507181080818497776e-03
            2.67008925352811121722851339654881e-01 7.04527723035169750609441052802140e-01 1.18276761225013507181080818497776e-03
            2.84633516120191137899197997285228e-02 2.67008925352811121722851339654881e-01 1.18276761225013507181080818497776e-03
            2.84633516120191137899197997285228e-02 7.04527723035169750609441052802140e-01 1.18276761225013507181080818497776e-03
            7.04527723035169750609441052802140e-01 2.67008925352811121722851339654881e-01 1.18276761225013507181080818497776e-03
            7.04527723035169750609441052802140e-01 2.84633516120191137899197997285228e-02 1.18276761225013507181080818497776e-03
            2.58604541010351018925916832813527e-01 2.22646709189586958915763048594272e-03 3.18088027822488136377165890422702e-04
            2.58604541010351018925916832813527e-01 7.39168991897753091535605562967248e-01 3.18088027822488136377165890422702e-04
            2.22646709189586958915763048594272e-03 2.58604541010351018925916832813527e-01 3.18088027822488136377165890422702e-04
            2.22646709189586958915763048594272e-03 7.39168991897753091535605562967248e-01 3.18088027822488136377165890422702e-04
            7.39168991897753091535605562967248e-01 2.58604541010351018925916832813527e-01 3.18088027822488136377165890422702e-04
            7.39168991897753091535605562967248e-01 2.22646709189586958915763048594272e-03 3.18088027822488136377165890422702e-04
            1.99624762677360428897443966889114e-01 1.20364369572249552131637528873398e-01 2.05838971587918350206192208418088e-03
            1.99624762677360428897443966889114e-01 6.80010867750389991215342888608575e-01 2.05838971587918350206192208418088e-03
            1.20364369572249552131637528873398e-01 1.99624762677360428897443966889114e-01 2.05838971587918350206192208418088e-03
            1.20364369572249552131637528873398e-01 6.80010867750389991215342888608575e-01 2.05838971587918350206192208418088e-03
            6.80010867750389991215342888608575e-01 1.99624762677360428897443966889114e-01 2.05838971587918350206192208418088e-03
            6.80010867750389991215342888608575e-01 1.20364369572249552131637528873398e-01 2.05838971587918350206192208418088e-03
            2.08890864384406460541043770717806e-01 1.64334461686715643802259023686929e-01 2.37178308508622943256982651405451e-03
            2.08890864384406460541043770717806e-01 6.26774673928877867901121589966351e-01 2.37178308508622943256982651405451e-03
            1.64334461686715643802259023686929e-01 2.08890864384406460541043770717806e-01 2.37178308508622943256982651405451e-03
            1.64334461686715643802259023686929e-01 6.26774673928877867901121589966351e-01 2.37178308508622943256982651405451e-03
            6.26774673928877867901121589966351e-01 2.08890864384406460541043770717806e-01 2.37178308508622943256982651405451e-03
            6.26774673928877867901121589966351e-01 1.64334461686715643802259023686929e-01 2.37178308508622943256982651405451e-03
            2.53333600601808928143299226576346e-01 1.23887458837676739387134716707806e-01 2.30113371257409425954532089519944e-03
            2.53333600601808928143299226576346e-01 6.22778940560514304713990441086935e-01 2.30113371257409425954532089519944e-03
            1.23887458837676739387134716707806e-01 2.53333600601808928143299226576346e-01 2.30113371257409425954532089519944e-03
            1.23887458837676739387134716707806e-01 6.22778940560514304713990441086935e-01 2.30113371257409425954532089519944e-03
            6.22778940560514304713990441086935e-01 2.53333600601808928143299226576346e-01 2.30113371257409425954532089519944e-03
            6.22778940560514304713990441086935e-01 1.23887458837676739387134716707806e-01 2.30113371257409425954532089519944e-03
            4.63796122746566341632501462299842e-01 1.60010389384323214144456581919940e-04 1.17504461240287803587382942094308e-04
            4.63796122746566341632501462299842e-01 5.36043866864049367570999038434820e-01 1.17504461240287803587382942094308e-04
            1.60010389384323214144456581919940e-04 4.63796122746566341632501462299842e-01 1.17504461240287803587382942094308e-04
            1.60010389384323214144456581919940e-04 5.36043866864049367570999038434820e-01 1.17504461240287803587382942094308e-04
            5.36043866864049367570999038434820e-01 4.63796122746566341632501462299842e-01 1.17504461240287803587382942094308e-04
            5.36043866864049367570999038434820e-01 1.60010389384323214144456581919940e-04 1.17504461240287803587382942094308e-04
            3.83717799050546426453678350299015e-01 1.08509201743662683412559866180658e-04 1.19936839465554535844038497316433e-04
            3.83717799050546426453678350299015e-01 6.16173691747709861843418366333935e-01 1.19936839465554535844038497316433e-04
            1.08509201743662683412559866180658e-04 3.83717799050546426453678350299015e-01 1.19936839465554535844038497316433e-04
            1.08509201743662683412559866180658e-04 6.16173691747709861843418366333935e-01 1.19936839465554535844038497316433e-04
            6.16173691747709861843418366333935e-01 3.83717799050546426453678350299015e-01 1.19936839465554535844038497316433e-04
            6.16173691747709861843418366333935e-01 1.08509201743662683412559866180658e-04 1.19936839465554535844038497316433e-04
            ];
        
    case  49
        
        % ALG. DEG.:   49
        % PTS CARD.:  435
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.2e-15
        
        xyw=[
            1.78373016229631531359700602479279e-01 1.78373016229631531359700602479279e-01 1.61493883722624081630825365607507e-03
            1.78373016229631531359700602479279e-01 6.43253967540736937280598795041442e-01 1.61493883722624081630825365607507e-03
            6.43253967540736937280598795041442e-01 1.78373016229631531359700602479279e-01 1.61493883722624081630825365607507e-03
            7.28406536060113435349094856974261e-02 7.28406536060113435349094856974261e-02 7.89545893009189385727819576743514e-04
            7.28406536060113435349094856974261e-02 8.54318692787977340685756644234061e-01 7.89545893009189385727819576743514e-04
            8.54318692787977340685756644234061e-01 7.28406536060113435349094856974261e-02 7.89545893009189385727819576743514e-04
            4.28988355639407015207353879304719e-01 4.28988355639407015207353879304719e-01 1.96545273941011594542871421253949e-03
            4.28988355639407015207353879304719e-01 1.42023288721185969585292241390562e-01 1.96545273941011594542871421253949e-03
            1.42023288721185969585292241390562e-01 4.28988355639407015207353879304719e-01 1.96545273941011594542871421253949e-03
            3.81168837845570263045402725765598e-01 3.81168837845570263045402725765598e-01 2.49792961248307800528611899437692e-03
            3.81168837845570263045402725765598e-01 2.37662324308859473909194548468804e-01 2.49792961248307800528611899437692e-03
            2.37662324308859473909194548468804e-01 3.81168837845570263045402725765598e-01 2.49792961248307800528611899437692e-03
            2.00738367597118549934265274714562e-01 2.00738367597118549934265274714562e-01 2.03895945143986153172410702438810e-03
            2.00738367597118549934265274714562e-01 5.98523264805762900131469450570876e-01 2.03895945143986153172410702438810e-03
            5.98523264805762900131469450570876e-01 2.00738367597118549934265274714562e-01 2.03895945143986153172410702438810e-03
            4.62782231758690903999919896705251e-02 4.62782231758690903999919896705251e-02 5.85893274137245004783147717120073e-04
            4.62782231758690903999919896705251e-02 9.07443553648261791444440405030036e-01 5.85893274137245004783147717120073e-04
            9.07443553648261791444440405030036e-01 4.62782231758690903999919896705251e-02 5.85893274137245004783147717120073e-04
            2.97143088863247917252863317116862e-01 2.97143088863247917252863317116862e-01 2.54202687899601894802592561006804e-03
            2.97143088863247917252863317116862e-01 4.05713822273504165494273365766276e-01 2.54202687899601894802592561006804e-03
            4.05713822273504165494273365766276e-01 2.97143088863247917252863317116862e-01 2.54202687899601894802592561006804e-03
            1.08421217690070140204916526727175e-02 1.08421217690070140204916526727175e-02 1.45496719895626378015254798192757e-04
            1.08421217690070140204916526727175e-02 9.78315756461985985836804502469022e-01 1.45496719895626378015254798192757e-04
            9.78315756461985985836804502469022e-01 1.08421217690070140204916526727175e-02 1.45496719895626378015254798192757e-04
            1.02853381663129031520220735274052e-01 1.02853381663129031520220735274052e-01 1.23399131936961087285808957858535e-03
            1.02853381663129031520220735274052e-01 7.94293236673741964715134145080810e-01 1.23399131936961087285808957858535e-03
            7.94293236673741964715134145080810e-01 1.02853381663129031520220735274052e-01 1.23399131936961087285808957858535e-03
            4.84173563416694874295131967301131e-01 4.84173563416694874295131967301131e-01 1.22377744593757582271253969707914e-03
            4.84173563416694874295131967301131e-01 3.16528731666102514097360653977375e-02 1.22377744593757582271253969707914e-03
            3.16528731666102514097360653977375e-02 4.84173563416694874295131967301131e-01 1.22377744593757582271253969707914e-03
            4.93261195566052690342218056684942e-01 4.93261195566052690342218056684942e-01 8.41735588392921113901068430607211e-04
            4.93261195566052690342218056684942e-01 1.34776088678946193155638866301160e-02 8.41735588392921113901068430607211e-04
            1.34776088678946193155638866301160e-02 4.93261195566052690342218056684942e-01 8.41735588392921113901068430607211e-04
            4.05732471975522424045834668504540e-01 4.05732471975522424045834668504540e-01 2.68271127030139796762475690172778e-03
            4.05732471975522424045834668504540e-01 1.88535056048955151908330662990920e-01 2.68271127030139796762475690172778e-03
            1.88535056048955151908330662990920e-01 4.05732471975522424045834668504540e-01 2.68271127030139796762475690172778e-03
            1.30421644925131446912658361725335e-01 1.30421644925131446912658361725335e-01 1.74963401588083019948249852149047e-03
            1.30421644925131446912658361725335e-01 7.39156710149737161685834507807158e-01 1.74963401588083019948249852149047e-03
            7.39156710149737161685834507807158e-01 1.30421644925131446912658361725335e-01 1.74963401588083019948249852149047e-03
            4.61434805630296263068146345176501e-01 4.61434805630296263068146345176501e-01 1.98000989575437390949330662692773e-03
            4.61434805630296263068146345176501e-01 7.71303887394074738637073096469976e-02 1.98000989575437390949330662692773e-03
            7.71303887394074738637073096469976e-02 4.61434805630296263068146345176501e-01 1.98000989575437390949330662692773e-03
            2.75739602431112201808005579550809e-02 2.75739602431112201808005579550809e-02 4.15975444451907095701204131188433e-04
            2.75739602431112201808005579550809e-02 9.44852079513777587393974499718752e-01 4.15975444451907095701204131188433e-04
            9.44852079513777587393974499718752e-01 2.75739602431112201808005579550809e-02 4.15975444451907095701204131188433e-04
            2.31842445375473937829724491166417e-01 2.31842445375473937829724491166417e-01 2.68220453544745730464771149570424e-03
            2.31842445375473937829724491166417e-01 5.36315109249052124340551017667167e-01 2.68220453544745730464771149570424e-03
            5.36315109249052124340551017667167e-01 2.31842445375473937829724491166417e-01 2.68220453544745730464771149570424e-03
            3.51721776627771431922298006611527e-01 3.51721776627771431922298006611527e-01 3.06338680377702494636582564169203e-03
            3.51721776627771431922298006611527e-01 2.96556446744457136155403986776946e-01 3.06338680377702494636582564169203e-03
            2.96556446744457136155403986776946e-01 3.51721776627771431922298006611527e-01 3.06338680377702494636582564169203e-03
            2.21242130998175424647089748475537e-03 2.21242130998175424647089748475537e-03 3.23057149542251866285863526329081e-05
            2.21242130998175424647089748475537e-03 9.95575157380036546150847698299913e-01 3.23057149542251866285863526329081e-05
            9.95575157380036546150847698299913e-01 2.21242130998175424647089748475537e-03 3.23057149542251866285863526329081e-05
            4.98789564691512810057361093640793e-01 4.98789564691512810057361093640793e-01 3.94277582726633988360193106004203e-04
            4.98789564691512810057361093640793e-01 2.42087061697437988527781271841377e-03 3.94277582726633988360193106004203e-04
            2.42087061697437988527781271841377e-03 4.98789564691512810057361093640793e-01 3.94277582726633988360193106004203e-04
            1.74709854307173090282034877418482e-01 4.00081550569362995317668207917450e-02 7.58974691375409980798050213479655e-04
            1.74709854307173090282034877418482e-01 7.85281990635890547736153166624717e-01 7.58974691375409980798050213479655e-04
            4.00081550569362995317668207917450e-02 1.74709854307173090282034877418482e-01 7.58974691375409980798050213479655e-04
            4.00081550569362995317668207917450e-02 7.85281990635890547736153166624717e-01 7.58974691375409980798050213479655e-04
            7.85281990635890547736153166624717e-01 1.74709854307173090282034877418482e-01 7.58974691375409980798050213479655e-04
            7.85281990635890547736153166624717e-01 4.00081550569362995317668207917450e-02 7.58974691375409980798050213479655e-04
            3.86965057163826176012122459724196e-01 8.66241723145184591703404919371678e-03 4.82018154443510245775461608985779e-04
            3.86965057163826176012122459724196e-01 6.04372525604721988479184346942930e-01 4.82018154443510245775461608985779e-04
            8.66241723145184591703404919371678e-03 3.86965057163826176012122459724196e-01 4.82018154443510245775461608985779e-04
            8.66241723145184591703404919371678e-03 6.04372525604721988479184346942930e-01 4.82018154443510245775461608985779e-04
            6.04372525604721988479184346942930e-01 3.86965057163826176012122459724196e-01 4.82018154443510245775461608985779e-04
            6.04372525604721988479184346942930e-01 8.66241723145184591703404919371678e-03 4.82018154443510245775461608985779e-04
            1.26060933435405392311068339949998e-01 1.30110914835385301988046613530514e-03 1.40675183266099687725766975709973e-04
            1.26060933435405392311068339949998e-01 8.72637957416240728214518185268389e-01 1.40675183266099687725766975709973e-04
            1.30110914835385301988046613530514e-03 1.26060933435405392311068339949998e-01 1.40675183266099687725766975709973e-04
            1.30110914835385301988046613530514e-03 8.72637957416240728214518185268389e-01 1.40675183266099687725766975709973e-04
            8.72637957416240728214518185268389e-01 1.26060933435405392311068339949998e-01 1.40675183266099687725766975709973e-04
            8.72637957416240728214518185268389e-01 1.30110914835385301988046613530514e-03 1.40675183266099687725766975709973e-04
            2.83924952000500496751556056551635e-01 2.45628505378345851628552054535248e-01 2.34311066873255612053039698139401e-03
            2.83924952000500496751556056551635e-01 4.70446542621153651619891888913116e-01 2.34311066873255612053039698139401e-03
            2.45628505378345851628552054535248e-01 2.83924952000500496751556056551635e-01 2.34311066873255612053039698139401e-03
            2.45628505378345851628552054535248e-01 4.70446542621153651619891888913116e-01 2.34311066873255612053039698139401e-03
            4.70446542621153651619891888913116e-01 2.83924952000500496751556056551635e-01 2.34311066873255612053039698139401e-03
            4.70446542621153651619891888913116e-01 2.45628505378345851628552054535248e-01 2.34311066873255612053039698139401e-03
            3.30248269334462274393615643930389e-01 2.50741636173609283577690121092019e-01 2.25894949919160613338586784948347e-03
            3.30248269334462274393615643930389e-01 4.19010094491928442028694234977593e-01 2.25894949919160613338586784948347e-03
            2.50741636173609283577690121092019e-01 3.30248269334462274393615643930389e-01 2.25894949919160613338586784948347e-03
            2.50741636173609283577690121092019e-01 4.19010094491928442028694234977593e-01 2.25894949919160613338586784948347e-03
            4.19010094491928442028694234977593e-01 3.30248269334462274393615643930389e-01 2.25894949919160613338586784948347e-03
            4.19010094491928442028694234977593e-01 2.50741636173609283577690121092019e-01 2.25894949919160613338586784948347e-03
            2.56045941068157811848493565776153e-01 1.50973749388043430690042878694612e-03 2.13206740649609125881971949389992e-04
            2.56045941068157811848493565776153e-01 7.42444321437961796128490732371574e-01 2.13206740649609125881971949389992e-04
            1.50973749388043430690042878694612e-03 2.56045941068157811848493565776153e-01 2.13206740649609125881971949389992e-04
            1.50973749388043430690042878694612e-03 7.42444321437961796128490732371574e-01 2.13206740649609125881971949389992e-04
            7.42444321437961796128490732371574e-01 2.56045941068157811848493565776153e-01 2.13206740649609125881971949389992e-04
            7.42444321437961796128490732371574e-01 1.50973749388043430690042878694612e-03 2.13206740649609125881971949389992e-04
            2.60395316134476195735203418735182e-01 1.08983380212925196062201393942814e-01 1.62341831505260721958683323151718e-03
            2.60395316134476195735203418735182e-01 6.30621303652598608202595187322004e-01 1.62341831505260721958683323151718e-03
            1.08983380212925196062201393942814e-01 2.60395316134476195735203418735182e-01 1.62341831505260721958683323151718e-03
            1.08983380212925196062201393942814e-01 6.30621303652598608202595187322004e-01 1.62341831505260721958683323151718e-03
            6.30621303652598608202595187322004e-01 2.60395316134476195735203418735182e-01 1.62341831505260721958683323151718e-03
            6.30621303652598608202595187322004e-01 1.08983380212925196062201393942814e-01 1.62341831505260721958683323151718e-03
            1.70712696379342027119818681057950e-01 1.01968716908776124707713961470290e-01 1.35567494687072057427124072148672e-03
            1.70712696379342027119818681057950e-01 7.27318586711881875928042973100673e-01 1.35567494687072057427124072148672e-03
            1.01968716908776124707713961470290e-01 1.70712696379342027119818681057950e-01 1.35567494687072057427124072148672e-03
            1.01968716908776124707713961470290e-01 7.27318586711881875928042973100673e-01 1.35567494687072057427124072148672e-03
            7.27318586711881875928042973100673e-01 1.70712696379342027119818681057950e-01 1.35567494687072057427124072148672e-03
            7.27318586711881875928042973100673e-01 1.01968716908776124707713961470290e-01 1.35567494687072057427124072148672e-03
            3.74644058095124909968376414326485e-01 1.97722885748598982413870572827364e-02 8.16379113471228505007071252208561e-04
            3.74644058095124909968376414326485e-01 6.05583653330015181381895672529936e-01 8.16379113471228505007071252208561e-04
            1.97722885748598982413870572827364e-02 3.74644058095124909968376414326485e-01 8.16379113471228505007071252208561e-04
            1.97722885748598982413870572827364e-02 6.05583653330015181381895672529936e-01 8.16379113471228505007071252208561e-04
            6.05583653330015181381895672529936e-01 3.74644058095124909968376414326485e-01 8.16379113471228505007071252208561e-04
            6.05583653330015181381895672529936e-01 1.97722885748598982413870572827364e-02 8.16379113471228505007071252208561e-04
            4.35168457293983068634446453870623e-01 1.35418528809368014266534263612130e-03 2.39515088089517383211635159767638e-04
            4.35168457293983068634446453870623e-01 5.63477357417923152560490507312352e-01 2.39515088089517383211635159767638e-04
            1.35418528809368014266534263612130e-03 4.35168457293983068634446453870623e-01 2.39515088089517383211635159767638e-04
            1.35418528809368014266534263612130e-03 5.63477357417923152560490507312352e-01 2.39515088089517383211635159767638e-04
            5.63477357417923152560490507312352e-01 4.35168457293983068634446453870623e-01 2.39515088089517383211635159767638e-04
            5.63477357417923152560490507312352e-01 1.35418528809368014266534263612130e-03 2.39515088089517383211635159767638e-04
            1.39092391229103939931377453831374e-01 8.76428525159974741720603219619079e-02 1.30013591471941371800757902832402e-03
            1.39092391229103939931377453831374e-01 7.73264756254898544263198800763348e-01 1.30013591471941371800757902832402e-03
            8.76428525159974741720603219619079e-02 1.39092391229103939931377453831374e-01 1.30013591471941371800757902832402e-03
            8.76428525159974741720603219619079e-02 7.73264756254898544263198800763348e-01 1.30013591471941371800757902832402e-03
            7.73264756254898544263198800763348e-01 1.39092391229103939931377453831374e-01 1.30013591471941371800757902832402e-03
            7.73264756254898544263198800763348e-01 8.76428525159974741720603219619079e-02 1.30013591471941371800757902832402e-03
            2.80150654301227930442763636165182e-01 7.84155789651995310807208738879126e-02 1.53690383100287290482621749276859e-03
            2.80150654301227930442763636165182e-01 6.41433766733572441332000835245708e-01 1.53690383100287290482621749276859e-03
            7.84155789651995310807208738879126e-02 2.80150654301227930442763636165182e-01 1.53690383100287290482621749276859e-03
            7.84155789651995310807208738879126e-02 6.41433766733572441332000835245708e-01 1.53690383100287290482621749276859e-03
            6.41433766733572441332000835245708e-01 2.80150654301227930442763636165182e-01 1.53690383100287290482621749276859e-03
            6.41433766733572441332000835245708e-01 7.84155789651995310807208738879126e-02 1.53690383100287290482621749276859e-03
            3.46331187427559528124731969001004e-01 6.27477726380917394388703201002500e-02 1.49390992511364570707876886501708e-03
            3.46331187427559528124731969001004e-01 5.90921039934348746314185518713202e-01 1.49390992511364570707876886501708e-03
            6.27477726380917394388703201002500e-02 3.46331187427559528124731969001004e-01 1.49390992511364570707876886501708e-03
            6.27477726380917394388703201002500e-02 5.90921039934348746314185518713202e-01 1.49390992511364570707876886501708e-03
            5.90921039934348746314185518713202e-01 3.46331187427559528124731969001004e-01 1.49390992511364570707876886501708e-03
            5.90921039934348746314185518713202e-01 6.27477726380917394388703201002500e-02 1.49390992511364570707876886501708e-03
            1.14244793358781793202183507673908e-01 7.10573045383911233024987197381961e-03 3.46295674839278632643063682294837e-04
            1.14244793358781793202183507673908e-01 8.78649476187379097069651834317483e-01 3.46295674839278632643063682294837e-04
            7.10573045383911233024987197381961e-03 1.14244793358781793202183507673908e-01 3.46295674839278632643063682294837e-04
            7.10573045383911233024987197381961e-03 8.78649476187379097069651834317483e-01 3.46295674839278632643063682294837e-04
            8.78649476187379097069651834317483e-01 1.14244793358781793202183507673908e-01 3.46295674839278632643063682294837e-04
            8.78649476187379097069651834317483e-01 7.10573045383911233024987197381961e-03 3.46295674839278632643063682294837e-04
            2.60147782714403918991052933051833e-01 4.08904742282701419431312217511731e-02 1.13209474794829221579173239575766e-03
            2.60147782714403918991052933051833e-01 6.98961743057325946004709749104222e-01 1.13209474794829221579173239575766e-03
            4.08904742282701419431312217511731e-02 2.60147782714403918991052933051833e-01 1.13209474794829221579173239575766e-03
            4.08904742282701419431312217511731e-02 6.98961743057325946004709749104222e-01 1.13209474794829221579173239575766e-03
            6.98961743057325946004709749104222e-01 2.60147782714403918991052933051833e-01 1.13209474794829221579173239575766e-03
            6.98961743057325946004709749104222e-01 4.08904742282701419431312217511731e-02 1.13209474794829221579173239575766e-03
            3.38595450369722106298553399028606e-01 3.24948830275603520179039662707510e-02 1.08418096863569413286498832604821e-03
            3.38595450369722106298553399028606e-01 6.28909666602717520866860922978958e-01 1.08418096863569413286498832604821e-03
            3.24948830275603520179039662707510e-02 3.38595450369722106298553399028606e-01 1.08418096863569413286498832604821e-03
            3.24948830275603520179039662707510e-02 6.28909666602717520866860922978958e-01 1.08418096863569413286498832604821e-03
            6.28909666602717520866860922978958e-01 3.38595450369722106298553399028606e-01 1.08418096863569413286498832604821e-03
            6.28909666602717520866860922978958e-01 3.24948830275603520179039662707510e-02 1.08418096863569413286498832604821e-03
            4.40524947160692070813325926792459e-01 8.72904428993618967946854780848298e-03 6.30167241237810705524469145899502e-04
            4.40524947160692070813325926792459e-01 5.50746008549371834916996704123449e-01 6.30167241237810705524469145899502e-04
            8.72904428993618967946854780848298e-03 4.40524947160692070813325926792459e-01 6.30167241237810705524469145899502e-04
            8.72904428993618967946854780848298e-03 5.50746008549371834916996704123449e-01 6.30167241237810705524469145899502e-04
            5.50746008549371834916996704123449e-01 4.40524947160692070813325926792459e-01 6.30167241237810705524469145899502e-04
            5.50746008549371834916996704123449e-01 8.72904428993618967946854780848298e-03 6.30167241237810705524469145899502e-04
            8.57234837407636202089022958716669e-02 1.57475866401919737010139943578224e-03 1.50964002721018871909161584987658e-04
            8.57234837407636202089022958716669e-02 9.12701757595217100238471630291315e-01 1.50964002721018871909161584987658e-04
            1.57475866401919737010139943578224e-03 8.57234837407636202089022958716669e-02 1.50964002721018871909161584987658e-04
            1.57475866401919737010139943578224e-03 9.12701757595217100238471630291315e-01 1.50964002721018871909161584987658e-04
            9.12701757595217100238471630291315e-01 8.57234837407636202089022958716669e-02 1.50964002721018871909161584987658e-04
            9.12701757595217100238471630291315e-01 1.57475866401919737010139943578224e-03 1.50964002721018871909161584987658e-04
            2.85984086639778534166822510087513e-01 2.01532205535708414112505693083222e-01 2.28081151372815354311240909623848e-03
            2.85984086639778534166822510087513e-01 5.12483707824513134987398643716006e-01 2.28081151372815354311240909623848e-03
            2.01532205535708414112505693083222e-01 2.85984086639778534166822510087513e-01 2.28081151372815354311240909623848e-03
            2.01532205535708414112505693083222e-01 5.12483707824513134987398643716006e-01 2.28081151372815354311240909623848e-03
            5.12483707824513134987398643716006e-01 2.85984086639778534166822510087513e-01 2.28081151372815354311240909623848e-03
            5.12483707824513134987398643716006e-01 2.01532205535708414112505693083222e-01 2.28081151372815354311240909623848e-03
            3.02243064852707965517453203574405e-01 5.14181635143590096137522493791039e-02 1.32157138195153366945489015193971e-03
            3.02243064852707965517453203574405e-01 6.46338771632933073441051874397090e-01 1.32157138195153366945489015193971e-03
            5.14181635143590096137522493791039e-02 3.02243064852707965517453203574405e-01 1.32157138195153366945489015193971e-03
            5.14181635143590096137522493791039e-02 6.46338771632933073441051874397090e-01 1.32157138195153366945489015193971e-03
            6.46338771632933073441051874397090e-01 3.02243064852707965517453203574405e-01 1.32157138195153366945489015193971e-03
            6.46338771632933073441051874397090e-01 5.14181635143590096137522493791039e-02 1.32157138195153366945489015193971e-03
            1.14557425728885831084369328891626e-01 1.90287488235584209550488310469518e-02 5.65729964641721269104357450174803e-04
            1.14557425728885831084369328891626e-01 8.66413825447555741021687936154194e-01 5.65729964641721269104357450174803e-04
            1.90287488235584209550488310469518e-02 1.14557425728885831084369328891626e-01 5.65729964641721269104357450174803e-04
            1.90287488235584209550488310469518e-02 8.66413825447555741021687936154194e-01 5.65729964641721269104357450174803e-04
            8.66413825447555741021687936154194e-01 1.14557425728885831084369328891626e-01 5.65729964641721269104357450174803e-04
            8.66413825447555741021687936154194e-01 1.90287488235584209550488310469518e-02 5.65729964641721269104357450174803e-04
            2.12296430019530718569242822013621e-01 4.07880592078401632560868961263623e-02 1.05443659419093794735178271082532e-03
            2.12296430019530718569242822013621e-01 7.46915510772629187563609320932301e-01 1.05443659419093794735178271082532e-03
            4.07880592078401632560868961263623e-02 2.12296430019530718569242822013621e-01 1.05443659419093794735178271082532e-03
            4.07880592078401632560868961263623e-02 7.46915510772629187563609320932301e-01 1.05443659419093794735178271082532e-03
            7.46915510772629187563609320932301e-01 2.12296430019530718569242822013621e-01 1.05443659419093794735178271082532e-03
            7.46915510772629187563609320932301e-01 4.07880592078401632560868961263623e-02 1.05443659419093794735178271082532e-03
            1.46528429709901986122133621393004e-01 2.85146118965765860142624887885177e-02 7.80409350560662233575737189994470e-04
            1.46528429709901986122133621393004e-01 8.24956958393521455619179505447391e-01 7.80409350560662233575737189994470e-04
            2.85146118965765860142624887885177e-02 1.46528429709901986122133621393004e-01 7.80409350560662233575737189994470e-04
            2.85146118965765860142624887885177e-02 8.24956958393521455619179505447391e-01 7.80409350560662233575737189994470e-04
            8.24956958393521455619179505447391e-01 1.46528429709901986122133621393004e-01 7.80409350560662233575737189994470e-04
            8.24956958393521455619179505447391e-01 2.85146118965765860142624887885177e-02 7.80409350560662233575737189994470e-04
            3.74081646108826992858098492433783e-01 1.53708658611640291979938410804607e-01 2.31542555620517639600186576842589e-03
            3.74081646108826992858098492433783e-01 4.72209695279532715161963096761610e-01 2.31542555620517639600186576842589e-03
            1.53708658611640291979938410804607e-01 3.74081646108826992858098492433783e-01 2.31542555620517639600186576842589e-03
            1.53708658611640291979938410804607e-01 4.72209695279532715161963096761610e-01 2.31542555620517639600186576842589e-03
            4.72209695279532715161963096761610e-01 3.74081646108826992858098492433783e-01 2.31542555620517639600186576842589e-03
            4.72209695279532715161963096761610e-01 1.53708658611640291979938410804607e-01 2.31542555620517639600186576842589e-03
            8.17068303997550704487196071568178e-02 2.46053203840888995435065567107813e-02 5.82418912261697192984943871607584e-04
            8.17068303997550704487196071568178e-02 8.93687849216156071641137259575771e-01 5.82418912261697192984943871607584e-04
            2.46053203840888995435065567107813e-02 8.17068303997550704487196071568178e-02 5.82418912261697192984943871607584e-04
            2.46053203840888995435065567107813e-02 8.93687849216156071641137259575771e-01 5.82418912261697192984943871607584e-04
            8.93687849216156071641137259575771e-01 8.17068303997550704487196071568178e-02 5.82418912261697192984943871607584e-04
            8.93687849216156071641137259575771e-01 2.46053203840888995435065567107813e-02 5.82418912261697192984943871607584e-04
            2.94147478880844692739060519670602e-01 2.35355019299825565792794179742486e-02 9.53923793556402197651999941285794e-04
            2.94147478880844692739060519670602e-01 6.82317019189172757620553966262378e-01 9.53923793556402197651999941285794e-04
            2.35355019299825565792794179742486e-02 2.94147478880844692739060519670602e-01 9.53923793556402197651999941285794e-04
            2.35355019299825565792794179742486e-02 6.82317019189172757620553966262378e-01 9.53923793556402197651999941285794e-04
            6.82317019189172757620553966262378e-01 2.94147478880844692739060519670602e-01 9.53923793556402197651999941285794e-04
            6.82317019189172757620553966262378e-01 2.35355019299825565792794179742486e-02 9.53923793556402197651999941285794e-04
            1.75070369482651533310502145468490e-01 1.37716721260120800174320265796268e-01 1.81036796651771916648321347764750e-03
            1.75070369482651533310502145468490e-01 6.87212909257227666515177588735241e-01 1.81036796651771916648321347764750e-03
            1.37716721260120800174320265796268e-01 1.75070369482651533310502145468490e-01 1.81036796651771916648321347764750e-03
            1.37716721260120800174320265796268e-01 6.87212909257227666515177588735241e-01 1.81036796651771916648321347764750e-03
            6.87212909257227666515177588735241e-01 1.75070369482651533310502145468490e-01 1.81036796651771916648321347764750e-03
            6.87212909257227666515177588735241e-01 1.37716721260120800174320265796268e-01 1.81036796651771916648321347764750e-03
            4.76205378057730083218856975690869e-02 1.09753100892318270626057952199517e-02 3.02352958305666897828589689822820e-04
            4.76205378057730083218856975690869e-02 9.41404152104995173289125887094997e-01 3.02352958305666897828589689822820e-04
            1.09753100892318270626057952199517e-02 4.76205378057730083218856975690869e-02 3.02352958305666897828589689822820e-04
            1.09753100892318270626057952199517e-02 9.41404152104995173289125887094997e-01 3.02352958305666897828589689822820e-04
            9.41404152104995173289125887094997e-01 4.76205378057730083218856975690869e-02 3.02352958305666897828589689822820e-04
            9.41404152104995173289125887094997e-01 1.09753100892318270626057952199517e-02 3.02352958305666897828589689822820e-04
            3.17572481513326898561189182146336e-01 1.60465657202724848451325101450493e-01 2.23578901441128224253374057184374e-03
            3.17572481513326898561189182146336e-01 5.21961861283948169720758869516430e-01 2.23578901441128224253374057184374e-03
            1.60465657202724848451325101450493e-01 3.17572481513326898561189182146336e-01 2.23578901441128224253374057184374e-03
            1.60465657202724848451325101450493e-01 5.21961861283948169720758869516430e-01 2.23578901441128224253374057184374e-03
            5.21961861283948169720758869516430e-01 3.17572481513326898561189182146336e-01 2.23578901441128224253374057184374e-03
            5.21961861283948169720758869516430e-01 1.60465657202724848451325101450493e-01 2.23578901441128224253374057184374e-03
            2.88138053995077414448644503863761e-01 1.32949267737699799996065053164784e-01 2.16089327797227810992741758866487e-03
            2.88138053995077414448644503863761e-01 5.78912678267222813310866058600368e-01 2.16089327797227810992741758866487e-03
            1.32949267737699799996065053164784e-01 2.88138053995077414448644503863761e-01 2.16089327797227810992741758866487e-03
            1.32949267737699799996065053164784e-01 5.78912678267222813310866058600368e-01 2.16089327797227810992741758866487e-03
            5.78912678267222813310866058600368e-01 2.88138053995077414448644503863761e-01 2.16089327797227810992741758866487e-03
            5.78912678267222813310866058600368e-01 1.32949267737699799996065053164784e-01 2.16089327797227810992741758866487e-03
            2.67970889303296155858902238833252e-01 8.75495123131142180539221442359121e-03 5.91722984001432023362654355480572e-04
            2.67970889303296155858902238833252e-01 7.23274159465392418866258594789542e-01 5.91722984001432023362654355480572e-04
            8.75495123131142180539221442359121e-03 2.67970889303296155858902238833252e-01 5.91722984001432023362654355480572e-04
            8.75495123131142180539221442359121e-03 7.23274159465392418866258594789542e-01 5.91722984001432023362654355480572e-04
            7.23274159465392418866258594789542e-01 2.67970889303296155858902238833252e-01 5.91722984001432023362654355480572e-04
            7.23274159465392418866258594789542e-01 8.75495123131142180539221442359121e-03 5.91722984001432023362654355480572e-04
            2.39379059203026878233089291825308e-01 2.07437619833565727034407899509461e-02 8.54923059629562084697773638453100e-04
            2.39379059203026878233089291825308e-01 7.39877178813616476205083927197848e-01 8.54923059629562084697773638453100e-04
            2.07437619833565727034407899509461e-02 2.39379059203026878233089291825308e-01 8.54923059629562084697773638453100e-04
            2.07437619833565727034407899509461e-02 7.39877178813616476205083927197848e-01 8.54923059629562084697773638453100e-04
            7.39877178813616476205083927197848e-01 2.39379059203026878233089291825308e-01 8.54923059629562084697773638453100e-04
            7.39877178813616476205083927197848e-01 2.07437619833565727034407899509461e-02 8.54923059629562084697773638453100e-04
            1.91853271569703082155555762255972e-01 1.97602896840749810147830345385955e-02 7.63033388765983148355198650847342e-04
            1.91853271569703082155555762255972e-01 7.88386438746221940299108155159047e-01 7.63033388765983148355198650847342e-04
            1.97602896840749810147830345385955e-02 1.91853271569703082155555762255972e-01 7.63033388765983148355198650847342e-04
            1.97602896840749810147830345385955e-02 7.88386438746221940299108155159047e-01 7.63033388765983148355198650847342e-04
            7.88386438746221940299108155159047e-01 1.91853271569703082155555762255972e-01 7.63033388765983148355198650847342e-04
            7.88386438746221940299108155159047e-01 1.97602896840749810147830345385955e-02 7.63033388765983148355198650847342e-04
            3.87324242534184959296794659167062e-01 4.29392857768670643814346021827077e-02 1.37645103473179263552095275002785e-03
            3.87324242534184959296794659167062e-01 5.69736471688947920810619507392403e-01 1.37645103473179263552095275002785e-03
            4.29392857768670643814346021827077e-02 3.87324242534184959296794659167062e-01 1.37645103473179263552095275002785e-03
            4.29392857768670643814346021827077e-02 5.69736471688947920810619507392403e-01 1.37645103473179263552095275002785e-03
            5.69736471688947920810619507392403e-01 3.87324242534184959296794659167062e-01 1.37645103473179263552095275002785e-03
            5.69736471688947920810619507392403e-01 4.29392857768670643814346021827077e-02 1.37645103473179263552095275002785e-03
            3.53532415828410828684269517907524e-01 1.17892418158290449015090928241989e-01 2.18161361839561964912070912703257e-03
            3.53532415828410828684269517907524e-01 5.28575166013298791689578592922771e-01 2.18161361839561964912070912703257e-03
            1.17892418158290449015090928241989e-01 3.53532415828410828684269517907524e-01 2.18161361839561964912070912703257e-03
            1.17892418158290449015090928241989e-01 5.28575166013298791689578592922771e-01 2.18161361839561964912070912703257e-03
            5.28575166013298791689578592922771e-01 3.53532415828410828684269517907524e-01 2.18161361839561964912070912703257e-03
            5.28575166013298791689578592922771e-01 1.17892418158290449015090928241989e-01 2.18161361839561964912070912703257e-03
            5.21774890148561099056934153850307e-02 2.53747294625837269133050710934185e-02 4.83142678116275741100754670398487e-04
            5.21774890148561099056934153850307e-02 9.22447781522560239508834456501063e-01 4.83142678116275741100754670398487e-04
            2.53747294625837269133050710934185e-02 5.21774890148561099056934153850307e-02 4.83142678116275741100754670398487e-04
            2.53747294625837269133050710934185e-02 9.22447781522560239508834456501063e-01 4.83142678116275741100754670398487e-04
            9.22447781522560239508834456501063e-01 5.21774890148561099056934153850307e-02 4.83142678116275741100754670398487e-04
            9.22447781522560239508834456501063e-01 2.53747294625837269133050710934185e-02 4.83142678116275741100754670398487e-04
            3.23702116151313945557888018811354e-01 9.35293820848460272232571810491208e-02 1.89077633403431978088848897101570e-03
            3.23702116151313945557888018811354e-01 5.82768501763839985585491376696154e-01 1.89077633403431978088848897101570e-03
            9.35293820848460272232571810491208e-02 3.23702116151313945557888018811354e-01 1.89077633403431978088848897101570e-03
            9.35293820848460272232571810491208e-02 5.82768501763839985585491376696154e-01 1.89077633403431978088848897101570e-03
            5.82768501763839985585491376696154e-01 3.23702116151313945557888018811354e-01 1.89077633403431978088848897101570e-03
            5.82768501763839985585491376696154e-01 9.35293820848460272232571810491208e-02 1.89077633403431978088848897101570e-03
            3.72059360674848627059674299744074e-01 1.98710348807312189509111099994243e-03 3.01784656373464839872189369174293e-04
            3.72059360674848627059674299744074e-01 6.25953535837078356429685754847014e-01 3.01784656373464839872189369174293e-04
            1.98710348807312189509111099994243e-03 3.72059360674848627059674299744074e-01 3.01784656373464839872189369174293e-04
            1.98710348807312189509111099994243e-03 6.25953535837078356429685754847014e-01 3.01784656373464839872189369174293e-04
            6.25953535837078356429685754847014e-01 3.72059360674848627059674299744074e-01 3.01784656373464839872189369174293e-04
            6.25953535837078356429685754847014e-01 1.98710348807312189509111099994243e-03 3.01784656373464839872189369174293e-04
            4.13692534607104922539377866996801e-01 1.08690196293505916824706503120979e-01 2.12758198041947772446991216099832e-03
            4.13692534607104922539377866996801e-01 4.77617269099389119002552206438850e-01 2.12758198041947772446991216099832e-03
            1.08690196293505916824706503120979e-01 4.13692534607104922539377866996801e-01 2.12758198041947772446991216099832e-03
            1.08690196293505916824706503120979e-01 4.77617269099389119002552206438850e-01 2.12758198041947772446991216099832e-03
            4.77617269099389119002552206438850e-01 4.13692534607104922539377866996801e-01 2.12758198041947772446991216099832e-03
            4.77617269099389119002552206438850e-01 1.08690196293505916824706503120979e-01 2.12758198041947772446991216099832e-03
            2.50865443641437146915507128142053e-01 1.74902334150963623127239543464384e-01 2.42452666120457566503731960949608e-03
            2.50865443641437146915507128142053e-01 5.74232222207599285468404559651390e-01 2.42452666120457566503731960949608e-03
            1.74902334150963623127239543464384e-01 2.50865443641437146915507128142053e-01 2.42452666120457566503731960949608e-03
            1.74902334150963623127239543464384e-01 5.74232222207599285468404559651390e-01 2.42452666120457566503731960949608e-03
            5.74232222207599285468404559651390e-01 2.50865443641437146915507128142053e-01 2.42452666120457566503731960949608e-03
            5.74232222207599285468404559651390e-01 1.74902334150963623127239543464384e-01 2.42452666120457566503731960949608e-03
            2.36847703538783593835148622019915e-01 6.69659133716484916698519214151020e-02 1.51274590766757783578955187664405e-03
            2.36847703538783593835148622019915e-01 6.96186383089567928372787264379440e-01 1.51274590766757783578955187664405e-03
            6.69659133716484916698519214151020e-02 2.36847703538783593835148622019915e-01 1.51274590766757783578955187664405e-03
            6.69659133716484916698519214151020e-02 6.96186383089567928372787264379440e-01 1.51274590766757783578955187664405e-03
            6.96186383089567928372787264379440e-01 2.36847703538783593835148622019915e-01 1.51274590766757783578955187664405e-03
            6.96186383089567928372787264379440e-01 6.69659133716484916698519214151020e-02 1.51274590766757783578955187664405e-03
            1.09957683042648218796344394831976e-01 4.09791337707322012029464985971572e-02 9.15132955474598562377319677096921e-04
            1.09957683042648218796344394831976e-01 8.49063183186619552245133490941953e-01 9.15132955474598562377319677096921e-04
            4.09791337707322012029464985971572e-02 1.09957683042648218796344394831976e-01 9.15132955474598562377319677096921e-04
            4.09791337707322012029464985971572e-02 8.49063183186619552245133490941953e-01 9.15132955474598562377319677096921e-04
            8.49063183186619552245133490941953e-01 1.09957683042648218796344394831976e-01 9.15132955474598562377319677096921e-04
            8.49063183186619552245133490941953e-01 4.09791337707322012029464985971572e-02 9.15132955474598562377319677096921e-04
            4.29731902895181538948321531279362e-01 2.44664150758457620005170696231289e-02 1.15564557263624014156466746072738e-03
            4.29731902895181538948321531279362e-01 5.45801682028972612314987600257155e-01 1.15564557263624014156466746072738e-03
            2.44664150758457620005170696231289e-02 4.29731902895181538948321531279362e-01 1.15564557263624014156466746072738e-03
            2.44664150758457620005170696231289e-02 5.45801682028972612314987600257155e-01 1.15564557263624014156466746072738e-03
            5.45801682028972612314987600257155e-01 4.29731902895181538948321531279362e-01 1.15564557263624014156466746072738e-03
            5.45801682028972612314987600257155e-01 2.44664150758457620005170696231289e-02 1.15564557263624014156466746072738e-03
            1.43768037433372491973315732138872e-01 5.66880883410084066009737568947457e-02 1.14880906414655133661317787385769e-03
            1.43768037433372491973315732138872e-01 7.99543874225619122242392222688068e-01 1.14880906414655133661317787385769e-03
            5.66880883410084066009737568947457e-02 1.43768037433372491973315732138872e-01 1.14880906414655133661317787385769e-03
            5.66880883410084066009737568947457e-02 7.99543874225619122242392222688068e-01 1.14880906414655133661317787385769e-03
            7.99543874225619122242392222688068e-01 1.43768037433372491973315732138872e-01 1.14880906414655133661317787385769e-03
            7.99543874225619122242392222688068e-01 5.66880883410084066009737568947457e-02 1.14880906414655133661317787385769e-03
            7.74106779986907661550432635522156e-02 9.65248336154636912831428219305963e-03 3.86871300727659520306972451564320e-04
            7.74106779986907661550432635522156e-02 9.12936838639762937575028445280623e-01 3.86871300727659520306972451564320e-04
            9.65248336154636912831428219305963e-03 7.74106779986907661550432635522156e-02 3.86871300727659520306972451564320e-04
            9.65248336154636912831428219305963e-03 9.12936838639762937575028445280623e-01 3.86871300727659520306972451564320e-04
            9.12936838639762937575028445280623e-01 7.74106779986907661550432635522156e-02 3.86871300727659520306972451564320e-04
            9.12936838639762937575028445280623e-01 9.65248336154636912831428219305963e-03 3.86871300727659520306972451564320e-04
            1.55706005665346897126610770101252e-01 1.10995676273211840018317886347177e-02 5.54555395331634269108500578937537e-04
            1.55706005665346897126610770101252e-01 8.33194426707331903259046157472767e-01 5.54555395331634269108500578937537e-04
            1.10995676273211840018317886347177e-02 1.55706005665346897126610770101252e-01 5.54555395331634269108500578937537e-04
            1.10995676273211840018317886347177e-02 8.33194426707331903259046157472767e-01 5.54555395331634269108500578937537e-04
            8.33194426707331903259046157472767e-01 1.55706005665346897126610770101252e-01 5.54555395331634269108500578937537e-04
            8.33194426707331903259046157472767e-01 1.10995676273211840018317886347177e-02 5.54555395331634269108500578937537e-04
            3.12853302449651204941716287066811e-01 1.97407178793647343148509598620421e-03 2.95864003539374426175051491227919e-04
            3.12853302449651204941716287066811e-01 6.85172625762412335070905555767240e-01 2.95864003539374426175051491227919e-04
            1.97407178793647343148509598620421e-03 3.12853302449651204941716287066811e-01 2.95864003539374426175051491227919e-04
            1.97407178793647343148509598620421e-03 6.85172625762412335070905555767240e-01 2.95864003539374426175051491227919e-04
            6.85172625762412335070905555767240e-01 3.12853302449651204941716287066811e-01 2.95864003539374426175051491227919e-04
            6.85172625762412335070905555767240e-01 1.97407178793647343148509598620421e-03 2.95864003539374426175051491227919e-04
            2.60902661381798156581268699483189e-02 1.10588672805732885634499851335022e-02 2.33164539169912336021711629641118e-04
            2.60902661381798156581268699483189e-02 9.62850866581246966902085659967270e-01 2.33164539169912336021711629641118e-04
            1.10588672805732885634499851335022e-02 2.60902661381798156581268699483189e-02 2.33164539169912336021711629641118e-04
            1.10588672805732885634499851335022e-02 9.62850866581246966902085659967270e-01 2.33164539169912336021711629641118e-04
            9.62850866581246966902085659967270e-01 2.60902661381798156581268699483189e-02 2.33164539169912336021711629641118e-04
            9.62850866581246966902085659967270e-01 1.10588672805732885634499851335022e-02 2.33164539169912336021711629641118e-04
            1.65672860817991568493923182359140e-01 2.46254442236619333975178314233290e-03 2.53027257107086016481301005143223e-04
            1.65672860817991568493923182359140e-01 8.31864594759642250743070235330379e-01 2.53027257107086016481301005143223e-04
            2.46254442236619333975178314233290e-03 1.65672860817991568493923182359140e-01 2.53027257107086016481301005143223e-04
            2.46254442236619333975178314233290e-03 8.31864594759642250743070235330379e-01 2.53027257107086016481301005143223e-04
            8.31864594759642250743070235330379e-01 1.65672860817991568493923182359140e-01 2.53027257107086016481301005143223e-04
            8.31864594759642250743070235330379e-01 2.46254442236619333975178314233290e-03 2.53027257107086016481301005143223e-04
            7.42257546763113590371219174812722e-02 4.63950129826373944275097471745539e-02 7.73102621795122239151121235067876e-04
            7.42257546763113590371219174812722e-02 8.79379232341051308985413470509229e-01 7.73102621795122239151121235067876e-04
            4.63950129826373944275097471745539e-02 7.42257546763113590371219174812722e-02 7.73102621795122239151121235067876e-04
            4.63950129826373944275097471745539e-02 8.79379232341051308985413470509229e-01 7.73102621795122239151121235067876e-04
            8.79379232341051308985413470509229e-01 7.42257546763113590371219174812722e-02 7.73102621795122239151121235067876e-04
            8.79379232341051308985413470509229e-01 4.63950129826373944275097471745539e-02 7.73102621795122239151121235067876e-04
            2.88339172370140305046160023039192e-02 2.07859992487419847906338077336841e-03 1.11908386263531061440706926291000e-04
            2.88339172370140305046160023039192e-02 9.69087482838111746730191953247413e-01 1.11908386263531061440706926291000e-04
            2.07859992487419847906338077336841e-03 2.88339172370140305046160023039192e-02 1.11908386263531061440706926291000e-04
            2.07859992487419847906338077336841e-03 9.69087482838111746730191953247413e-01 1.11908386263531061440706926291000e-04
            9.69087482838111746730191953247413e-01 2.88339172370140305046160023039192e-02 1.11908386263531061440706926291000e-04
            9.69087482838111746730191953247413e-01 2.07859992487419847906338077336841e-03 1.11908386263531061440706926291000e-04
            1.86822660672315066232584968020092e-01 6.83532967983220413055178710237669e-02 1.47809259194698369536957738290539e-03
            1.86822660672315066232584968020092e-01 7.44824042529362850828533737512771e-01 1.47809259194698369536957738290539e-03
            6.83532967983220413055178710237669e-02 1.86822660672315066232584968020092e-01 1.47809259194698369536957738290539e-03
            6.83532967983220413055178710237669e-02 7.44824042529362850828533737512771e-01 1.47809259194698369536957738290539e-03
            7.44824042529362850828533737512771e-01 1.86822660672315066232584968020092e-01 1.47809259194698369536957738290539e-03
            7.44824042529362850828533737512771e-01 6.83532967983220413055178710237669e-02 1.47809259194698369536957738290539e-03
            3.27083928185687322631736151379300e-01 1.06763043029804560041240435452892e-02 7.14520924585107107714077034898992e-04
            3.27083928185687322631736151379300e-01 6.62239767511332177996052905655233e-01 7.14520924585107107714077034898992e-04
            1.06763043029804560041240435452892e-02 3.27083928185687322631736151379300e-01 7.14520924585107107714077034898992e-04
            1.06763043029804560041240435452892e-02 6.62239767511332177996052905655233e-01 7.14520924585107107714077034898992e-04
            6.62239767511332177996052905655233e-01 3.27083928185687322631736151379300e-01 7.14520924585107107714077034898992e-04
            6.62239767511332177996052905655233e-01 1.06763043029804560041240435452892e-02 7.14520924585107107714077034898992e-04
            1.03891103433826778590365336185641e-01 6.91692834636133124792678472658736e-02 1.07293467420879144041956365640544e-03
            1.03891103433826778590365336185641e-01 8.26939613102559922808154624362942e-01 1.07293467420879144041956365640544e-03
            6.91692834636133124792678472658736e-02 1.03891103433826778590365336185641e-01 1.07293467420879144041956365640544e-03
            6.91692834636133124792678472658736e-02 8.26939613102559922808154624362942e-01 1.07293467420879144041956365640544e-03
            8.26939613102559922808154624362942e-01 1.03891103433826778590365336185641e-01 1.07293467420879144041956365640544e-03
            8.26939613102559922808154624362942e-01 6.91692834636133124792678472658736e-02 1.07293467420879144041956365640544e-03
            2.15534288242954619185809406189946e-01 1.00668809434643721778535052635561e-01 1.79238370972849346378363843967918e-03
            2.15534288242954619185809406189946e-01 6.83796902322401645157867733360035e-01 1.79238370972849346378363843967918e-03
            1.00668809434643721778535052635561e-01 2.15534288242954619185809406189946e-01 1.79238370972849346378363843967918e-03
            1.00668809434643721778535052635561e-01 6.83796902322401645157867733360035e-01 1.79238370972849346378363843967918e-03
            6.83796902322401645157867733360035e-01 2.15534288242954619185809406189946e-01 1.79238370972849346378363843967918e-03
            6.83796902322401645157867733360035e-01 1.00668809434643721778535052635561e-01 1.79238370972849346378363843967918e-03
            3.43151613027389945642653401591815e-01 2.01217937528444273942440645441820e-01 2.72636788911361036685798708845141e-03
            3.43151613027389945642653401591815e-01 4.55630449444165752659330337337451e-01 2.72636788911361036685798708845141e-03
            2.01217937528444273942440645441820e-01 3.43151613027389945642653401591815e-01 2.72636788911361036685798708845141e-03
            2.01217937528444273942440645441820e-01 4.55630449444165752659330337337451e-01 2.72636788911361036685798708845141e-03
            4.55630449444165752659330337337451e-01 3.43151613027389945642653401591815e-01 2.72636788911361036685798708845141e-03
            4.55630449444165752659330337337451e-01 2.01217937528444273942440645441820e-01 2.72636788911361036685798708845141e-03
            2.22408758723325439277118675818201e-01 1.43453248492636881161743644952367e-01 2.19083288782857691159855484386298e-03
            2.22408758723325439277118675818201e-01 6.34137992784037707316713294858346e-01 2.19083288782857691159855484386298e-03
            1.43453248492636881161743644952367e-01 2.22408758723325439277118675818201e-01 2.19083288782857691159855484386298e-03
            1.43453248492636881161743644952367e-01 6.34137992784037707316713294858346e-01 2.19083288782857691159855484386298e-03
            6.34137992784037707316713294858346e-01 2.22408758723325439277118675818201e-01 2.19083288782857691159855484386298e-03
            6.34137992784037707316713294858346e-01 1.43453248492636881161743644952367e-01 2.19083288782857691159855484386298e-03
            5.34809769143942265179347828052414e-02 2.12516053769432446476916886979325e-03 1.53587267982227722307175810101398e-04
            5.34809769143942265179347828052414e-02 9.44393862547911466798211677087238e-01 1.53587267982227722307175810101398e-04
            2.12516053769432446476916886979325e-03 5.34809769143942265179347828052414e-02 1.53587267982227722307175810101398e-04
            2.12516053769432446476916886979325e-03 9.44393862547911466798211677087238e-01 1.53587267982227722307175810101398e-04
            9.44393862547911466798211677087238e-01 5.34809769143942265179347828052414e-02 1.53587267982227722307175810101398e-04
            9.44393862547911466798211677087238e-01 2.12516053769432446476916886979325e-03 1.53587267982227722307175810101398e-04
            4.38788968845026938758735468582017e-01 5.14225024494020452325848680175113e-02 1.61908850822945577993539068017981e-03
            4.38788968845026938758735468582017e-01 5.09788528705571009069785759493243e-01 1.61908850822945577993539068017981e-03
            5.14225024494020452325848680175113e-02 4.38788968845026938758735468582017e-01 1.61908850822945577993539068017981e-03
            5.14225024494020452325848680175113e-02 5.09788528705571009069785759493243e-01 1.61908850822945577993539068017981e-03
            5.09788528705571009069785759493243e-01 4.38788968845026938758735468582017e-01 1.61908850822945577993539068017981e-03
            5.09788528705571009069785759493243e-01 5.14225024494020452325848680175113e-02 1.61908850822945577993539068017981e-03
            3.91932101375555841382691824037465e-01 7.81526329255782525873286203932366e-02 2.05269142946377242120448158857471e-03
            3.91932101375555841382691824037465e-01 5.29915265698865822763252708682558e-01 2.05269142946377242120448158857471e-03
            7.81526329255782525873286203932366e-02 3.91932101375555841382691824037465e-01 2.05269142946377242120448158857471e-03
            7.81526329255782525873286203932366e-02 5.29915265698865822763252708682558e-01 2.05269142946377242120448158857471e-03
            5.29915265698865822763252708682558e-01 3.91932101375555841382691824037465e-01 2.05269142946377242120448158857471e-03
            5.29915265698865822763252708682558e-01 7.81526329255782525873286203932366e-02 2.05269142946377242120448158857471e-03
            2.11082179897034566407043598701421e-01 6.49692444592179802614895223200620e-03 5.34179366171051456940732649059100e-04
            2.11082179897034566407043598701421e-01 7.82420895657043646842510042915819e-01 5.34179366171051456940732649059100e-04
            6.49692444592179802614895223200620e-03 2.11082179897034566407043598701421e-01 5.34179366171051456940732649059100e-04
            6.49692444592179802614895223200620e-03 7.82420895657043646842510042915819e-01 5.34179366171051456940732649059100e-04
            7.82420895657043646842510042915819e-01 2.11082179897034566407043598701421e-01 5.34179366171051456940732649059100e-04
            7.82420895657043646842510042915819e-01 6.49692444592179802614895223200620e-03 5.34179366171051456940732649059100e-04
            1.17098573109517579970972889213954e-02 2.09030641915285923110312538142352e-03 7.11599458074914901790225529865097e-05
            1.17098573109517579970972889213954e-02 9.86199836269895380169714371731970e-01 7.11599458074914901790225529865097e-05
            2.09030641915285923110312538142352e-03 1.17098573109517579970972889213954e-02 7.11599458074914901790225529865097e-05
            2.09030641915285923110312538142352e-03 9.86199836269895380169714371731970e-01 7.11599458074914901790225529865097e-05
            9.86199836269895380169714371731970e-01 1.17098573109517579970972889213954e-02 7.11599458074914901790225529865097e-05
            9.86199836269895380169714371731970e-01 2.09030641915285923110312538142352e-03 7.11599458074914901790225529865097e-05
            2.06383353759989840447985898208572e-01 1.39863518147532501665827875392267e-04 8.02792352761152354975043854423689e-05
            2.06383353759989840447985898208572e-01 7.93476782721862616831742798240157e-01 8.02792352761152354975043854423689e-05
            1.39863518147532501665827875392267e-04 2.06383353759989840447985898208572e-01 8.02792352761152354975043854423689e-05
            1.39863518147532501665827875392267e-04 7.93476782721862616831742798240157e-01 8.02792352761152354975043854423689e-05
            7.93476782721862616831742798240157e-01 2.06383353759989840447985898208572e-01 8.02792352761152354975043854423689e-05
            7.93476782721862616831742798240157e-01 1.39863518147532501665827875392267e-04 8.02792352761152354975043854423689e-05
            ];
        
    case  50
        
        % ALG. DEG.:   50
        % PTS CARD.:  453
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.4e-15
        
        xyw=[
            2.56355863384790827463888263082481e-01 2.56355863384790827463888263082481e-01 1.66434758735661478268319513063034e-03
            2.56355863384790827463888263082481e-01 4.87288273230418345072223473835038e-01 1.66434758735661478268319513063034e-03
            4.87288273230418345072223473835038e-01 2.56355863384790827463888263082481e-01 1.66434758735661478268319513063034e-03
            1.63057814214604269824338000915986e-03 1.63057814214604269824338000915986e-03 1.82966121231848889868137203462695e-05
            1.63057814214604269824338000915986e-03 9.96738843715707867332298519613687e-01 1.82966121231848889868137203462695e-05
            9.96738843715707867332298519613687e-01 1.63057814214604269824338000915986e-03 1.82966121231848889868137203462695e-05
            4.90326869248451202576433161084424e-01 4.90326869248451202576433161084424e-01 8.25698178639569426578992406007274e-04
            4.90326869248451202576433161084424e-01 1.93462615030975948471336778311525e-02 8.25698178639569426578992406007274e-04
            1.93462615030975948471336778311525e-02 4.90326869248451202576433161084424e-01 8.25698178639569426578992406007274e-04
            4.22587168507520649640696319693234e-01 4.22587168507520649640696319693234e-01 2.12954833909573563663353290564828e-03
            4.22587168507520649640696319693234e-01 1.54825662984958700718607360613532e-01 2.12954833909573563663353290564828e-03
            1.54825662984958700718607360613532e-01 4.22587168507520649640696319693234e-01 2.12954833909573563663353290564828e-03
            4.80575676434030041317413406432024e-01 4.80575676434030041317413406432024e-01 1.21213997870494682822239962405320e-03
            4.80575676434030041317413406432024e-01 3.88486471319399173651731871359516e-02 1.21213997870494682822239962405320e-03
            3.88486471319399173651731871359516e-02 4.80575676434030041317413406432024e-01 1.21213997870494682822239962405320e-03
            4.59586771824411877140903470717603e-01 4.59586771824411877140903470717603e-01 1.79135560553105358493064525049476e-03
            4.59586771824411877140903470717603e-01 8.08264563511762457181930585647933e-02 1.79135560553105358493064525049476e-03
            8.08264563511762457181930585647933e-02 4.59586771824411877140903470717603e-01 1.79135560553105358493064525049476e-03
            1.13333234135074248372099248172162e-01 1.13333234135074248372099248172162e-01 1.38979940022233156016184807413083e-03
            1.13333234135074248372099248172162e-01 7.73333531729851531011377119284589e-01 1.38979940022233156016184807413083e-03
            7.73333531729851531011377119284589e-01 1.13333234135074248372099248172162e-01 1.38979940022233156016184807413083e-03
            1.09390130646629217536780487307624e-02 1.09390130646629217536780487307624e-02 1.49976093457403371464117802780436e-04
            1.09390130646629217536780487307624e-02 9.78121973870674121798174383002333e-01 1.49976093457403371464117802780436e-04
            9.78121973870674121798174383002333e-01 1.09390130646629217536780487307624e-02 1.49976093457403371464117802780436e-04
            7.86463619513996653465071062782954e-02 7.86463619513996653465071062782954e-02 1.03393583353988721858229826722209e-03
            7.86463619513996653465071062782954e-02 8.42707276097200641551410171814496e-01 1.03393583353988721858229826722209e-03
            8.42707276097200641551410171814496e-01 7.86463619513996653465071062782954e-02 1.03393583353988721858229826722209e-03
            1.70258882364131336473178635060322e-01 1.70258882364131336473178635060322e-01 2.01385549425915589757951629223953e-03
            1.70258882364131336473178635060322e-01 6.59482235271737327053642729879357e-01 2.01385549425915589757951629223953e-03
            6.59482235271737327053642729879357e-01 1.70258882364131336473178635060322e-01 2.01385549425915589757951629223953e-03
            1.99633459336817115081785800612124e-01 1.99633459336817115081785800612124e-01 2.28533694456309029521823283914728e-03
            1.99633459336817115081785800612124e-01 6.00733081326365825347579630033579e-01 2.28533694456309029521823283914728e-03
            6.00733081326365825347579630033579e-01 1.99633459336817115081785800612124e-01 2.28533694456309029521823283914728e-03
            4.00668932548830014628293838541140e-01 4.00668932548830014628293838541140e-01 2.58267796372693703155221101042116e-03
            4.00668932548830014628293838541140e-01 1.98662134902339970743412322917720e-01 2.58267796372693703155221101042116e-03
            1.98662134902339970743412322917720e-01 4.00668932548830014628293838541140e-01 2.58267796372693703155221101042116e-03
            1.40613874056384147159803887916496e-01 1.40613874056384147159803887916496e-01 1.74263167672632853129333074804208e-03
            1.40613874056384147159803887916496e-01 7.18772251887231705680392224167008e-01 1.74263167672632853129333074804208e-03
            7.18772251887231705680392224167008e-01 1.40613874056384147159803887916496e-01 1.74263167672632853129333074804208e-03
            3.51190364445415514893511499394663e-01 3.511