
function [xyw_bar,pointset_stats]=set_zhang_cui_liu_barycentric(deg)

%--------------------------------------------------------------------
% CUBATURE RULE ON THE SIMPLEX [0 0; 0 1; 1 0] WITH ADE EQUAL TO deg.
% THE CUBATURE RULE IS DESCRIBED IN BARYCENTRICAL FORM.
%--------------------------------------------------------------------
% INPUT:
% deg: algebraic degree of precision (ADE) of the rule.
%--------------------------------------------------------------------
% OUTPUT:
% xyw_bar: MATRIX OF NODES xyw(:,1:3) IN BARYCENTRICAL COOR. AND WEIGHTS xyw(:,4).
%      REMARK: The sum of the weights makes 1.
%
% pointset_stats:
%    statistics of the pointset stored in a matrix. Each rows contains
%    degree, cardinality of the full pointset, moment error w.r.t.
%    orthn. Dubiner basis, number of points with negative weights,
%    number of points outside the domain.
%--------------------------------------------------------------------
% SOURCE:
% L. Zhang, T. Cui, H. Liu
% "A set of Symmetric Quadrature rules on triangles and tetrahedra"
% J. Comput. Math. 27, No. I, (2009), pp.89-96.
%--------------------------------------------------------------------
% | DEG | CARD | W- | PT.O|   M.E.   |Q.|
%--------------------------------------------------------------------
% |   1 |   1 |   0 |   0 | 1.92e-16 |PI|
% |   2 |   3 |   0 |   0 | 3.68e-16 |PI|
% |   3 |   6 |   0 |   0 | 1.46e-16 |PI|
% |   4 |   6 |   0 |   0 | 1.39e-16 |PI|
% |   5 |   7 |   0 |   0 | 1.98e-16 |PI|
% |   6 |  12 |   0 |   0 | 2.36e-16 |PI|
% |   7 |  16 |   0 |   0 | 3.75e-16 |PI|
% |   8 |  21 |   0 |   0 | 4.42e-13 |PI|
% |   9 |  19 |   0 |   0 | 5.55e-16 |PI|
% |  10 |  25 |   0 |   0 | 4.72e-16 |PI|
% |  11 |  28 |   0 |   0 | 8.33e-16 |PI|
% |  12 |  33 |   0 |   0 | 4.82e-16 |PI|
% |  13 |  37 |   0 |   0 | 5.83e-16 |PI|
% |  14 |  42 |   0 |   0 | 4.58e-16 |PI|
% |  15 |  49 |   0 |   0 | 4.82e-16 |PI|
% |  16 |  55 |   0 |   0 | 8.12e-16 |PI|
% |  17 |  60 |   0 |   0 | 6.21e-16 |PI|
% |  18 |  73 |   0 |   0 | 1.01e-15 |PI|
% |  20 |  82 |   0 |   0 | 9.06e-16 |PI|
% |  21 |  87 |   0 |   0 | 5.59e-16 |PI|
% |  22 |  96 |   0 |   0 | 6.66e-16 |PI|
% |  24 | 112 |   0 |   0 | 1.34e-15 |PI|
% |  25 | 126 |   0 |   0 | 6.17e-16 |PI|
% |  26 | 133 |   0 |   0 | 8.50e-16 |PI|
% |  27 | 145 |   0 |   0 | 9.99e-16 |PI|
% |  28 | 154 |   0 |   0 | 1.11e-15 |PI|
% |  29 | 166 |   0 |   0 | 1.33e-15 |PI|
%--------------------------------------------------------------------
% LEGEND:
%--------------------------------------------------------------------
% DEG  : algebraic degree of precision
% CARD : cardinality of the pointset
% W-   : negative weights
% PT.O : points outside the simplex [0 0; 1 0; 0 1].
% M.E. : maximum moment error in inf norm.
% Q.: quality of the rule.
%        PI: pos. weights, points in simplex
%        NI: some neg. weights, points in simplex
%        P0: pos. weights, some points out of the simplex
%        N0: some neg. weights, some points out of the simplex
%--------------------------------------------------------------------


switch deg
    
    case   1
        
        % ALG. DEG.:    1
        % PTS CARD.:    1
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.9e-16
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 1.00000000000000000000000000000000e+00
            ];
        
    case   2
        
        % ALG. DEG.:    2
        % PTS CARD.:    3
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.7e-16
        
        xyw_bar=[
            1.66666666666666657414808128123695e-01 1.66666666666666657414808128123695e-01 6.66666666666666740681534975010436e-01 3.33333333333333314829616256247391e-01
            1.66666666666666657414808128123695e-01 6.66666666666666740681534975010436e-01 1.66666666666666657414808128123695e-01 3.33333333333333314829616256247391e-01
            6.66666666666666740681534975010436e-01 1.66666666666666657414808128123695e-01 1.66666666666666657414808128123695e-01 3.33333333333333314829616256247391e-01
            ];
        
    case   3
        
        % ALG. DEG.:    3
        % PTS CARD.:    6
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.5e-16
        
        xyw_bar=[
            1.62882850395891898287459298444446e-01 1.62882850395891898287459298444446e-01 6.74234299208216203425081403111108e-01 2.81149802440979645901109051919775e-01
            1.62882850395891898287459298444446e-01 6.74234299208216203425081403111108e-01 1.62882850395891898287459298444446e-01 2.81149802440979645901109051919775e-01
            6.74234299208216203425081403111108e-01 1.62882850395891898287459298444446e-01 1.62882850395891898287459298444446e-01 2.81149802440979645901109051919775e-01
            4.77919883567563708037795322525199e-01 4.77919883567563708037795322525199e-01 4.41602328648725839244093549496029e-02 5.21835308923536828062950121420727e-02
            4.77919883567563708037795322525199e-01 4.41602328648725839244093549496029e-02 4.77919883567563708037795322525199e-01 5.21835308923536828062950121420727e-02
            4.41602328648725839244093549496029e-02 4.77919883567563708037795322525199e-01 4.77919883567563708037795322525199e-01 5.21835308923536828062950121420727e-02
            ];
        
    case   4
        
        % ALG. DEG.:    4
        % PTS CARD.:    6
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.4e-16
        
        xyw_bar=[
            4.45948490915964890213274429697776e-01 4.45948490915964890213274429697776e-01 1.08103018168070219573451140604448e-01 2.23381589678011471811203136894619e-01
            4.45948490915964890213274429697776e-01 1.08103018168070219573451140604448e-01 4.45948490915964890213274429697776e-01 2.23381589678011471811203136894619e-01
            1.08103018168070219573451140604448e-01 4.45948490915964890213274429697776e-01 4.45948490915964890213274429697776e-01 2.23381589678011471811203136894619e-01
            9.15762135097707430375635340169538e-02 9.15762135097707430375635340169538e-02 8.16847572980458513924872931966092e-01 1.09951743655321870773988734981685e-01
            9.15762135097707430375635340169538e-02 8.16847572980458513924872931966092e-01 9.15762135097707430375635340169538e-02 1.09951743655321870773988734981685e-01
            8.16847572980458513924872931966092e-01 9.15762135097707430375635340169538e-02 9.15762135097707430375635340169538e-02 1.09951743655321870773988734981685e-01
            ];
        
    case   5
        
        % ALG. DEG.:    5
        % PTS CARD.:    7
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 2.0e-16
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 2.25000000000000005551115123125783e-01
            1.01286507323456342888334802410100e-01 1.01286507323456342888334802410100e-01 7.97426985353087314223330395179801e-01 1.25939180544827139529573400977824e-01
            1.01286507323456342888334802410100e-01 7.97426985353087314223330395179801e-01 1.01286507323456342888334802410100e-01 1.25939180544827139529573400977824e-01
            7.97426985353087314223330395179801e-01 1.01286507323456342888334802410100e-01 1.01286507323456342888334802410100e-01 1.25939180544827139529573400977824e-01
            4.70142064105115109473587153843255e-01 4.70142064105115109473587153843255e-01 5.97158717897697810528256923134904e-02 1.32394152788506191953388224646915e-01
            4.70142064105115109473587153843255e-01 5.97158717897697810528256923134904e-02 4.70142064105115109473587153843255e-01 1.32394152788506191953388224646915e-01
            5.97158717897697810528256923134904e-02 4.70142064105115109473587153843255e-01 4.70142064105115109473587153843255e-01 1.32394152788506191953388224646915e-01
            ];
        
    case   6
        
        % ALG. DEG.:    6
        % PTS CARD.:   12
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 2.4e-16
        
        xyw_bar=[
            6.30890144915022266225435032538371e-02 6.30890144915022266225435032538371e-02 8.73821971016995546754912993492326e-01 5.08449063702068326797700592578622e-02
            6.30890144915022266225435032538371e-02 8.73821971016995546754912993492326e-01 6.30890144915022266225435032538371e-02 5.08449063702068326797700592578622e-02
            8.73821971016995546754912993492326e-01 6.30890144915022266225435032538371e-02 6.30890144915022266225435032538371e-02 5.08449063702068326797700592578622e-02
            2.49286745170910428726074314909056e-01 2.49286745170910428726074314909056e-01 5.01426509658179142547851370181888e-01 1.16786275726379396022736045779311e-01
            2.49286745170910428726074314909056e-01 5.01426509658179142547851370181888e-01 2.49286745170910428726074314909056e-01 1.16786275726379396022736045779311e-01
            5.01426509658179142547851370181888e-01 2.49286745170910428726074314909056e-01 2.49286745170910428726074314909056e-01 1.16786275726379396022736045779311e-01
            5.31450498448169453280520713178703e-02 3.10352451033784393352732422499685e-01 6.36502499121398668258109410089673e-01 8.28510756183735846969184990484791e-02
            5.31450498448169453280520713178703e-02 6.36502499121398668258109410089673e-01 3.10352451033784393352732422499685e-01 8.28510756183735846969184990484791e-02
            3.10352451033784393352732422499685e-01 5.31450498448169453280520713178703e-02 6.36502499121398668258109410089673e-01 8.28510756183735846969184990484791e-02
            3.10352451033784393352732422499685e-01 6.36502499121398668258109410089673e-01 5.31450498448169453280520713178703e-02 8.28510756183735846969184990484791e-02
            6.36502499121398668258109410089673e-01 5.31450498448169453280520713178703e-02 3.10352451033784393352732422499685e-01 8.28510756183735846969184990484791e-02
            6.36502499121398668258109410089673e-01 3.10352451033784393352732422499685e-01 5.31450498448169453280520713178703e-02 8.28510756183735846969184990484791e-02
            ];
        
    case   7
        
        % ALG. DEG.:    7
        % PTS CARD.:   16
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.7e-16
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 1.44315607677787172136163462710101e-01
            1.70569307751760212976677166807349e-01 1.70569307751760212976677166807349e-01 6.58861384496479574046645666385302e-01 1.03217370534718244634575512463925e-01
            1.70569307751760212976677166807349e-01 6.58861384496479574046645666385302e-01 1.70569307751760212976677166807349e-01 1.03217370534718244634575512463925e-01
            6.58861384496479574046645666385302e-01 1.70569307751760212976677166807349e-01 1.70569307751760212976677166807349e-01 1.03217370534718244634575512463925e-01
            5.05472283170309774624762155781355e-02 5.05472283170309774624762155781355e-02 8.98905543365938086708410992287099e-01 3.24584976231980792960030157701112e-02
            5.05472283170309774624762155781355e-02 8.98905543365938086708410992287099e-01 5.05472283170309774624762155781355e-02 3.24584976231980792960030157701112e-02
            8.98905543365938086708410992287099e-01 5.05472283170309774624762155781355e-02 5.05472283170309774624762155781355e-02 3.24584976231980792960030157701112e-02
            4.59292588292723180654064663031022e-01 4.59292588292723180654064663031022e-01 8.14148234145536386918706739379559e-02 9.50916342672846193195823616406415e-02
            4.59292588292723180654064663031022e-01 8.14148234145536386918706739379559e-02 4.59292588292723180654064663031022e-01 9.50916342672846193195823616406415e-02
            8.14148234145536386918706739379559e-02 4.59292588292723180654064663031022e-01 4.59292588292723180654064663031022e-01 9.50916342672846193195823616406415e-02
            2.63112829634638112352718053443823e-01 8.39477740995760515763102205255564e-03 7.28492392955404244325734453013865e-01 2.72303141744349927466650740370824e-02
            2.63112829634638112352718053443823e-01 7.28492392955404244325734453013865e-01 8.39477740995760515763102205255564e-03 2.72303141744349927466650740370824e-02
            8.39477740995760515763102205255564e-03 2.63112829634638112352718053443823e-01 7.28492392955404244325734453013865e-01 2.72303141744349927466650740370824e-02
            8.39477740995760515763102205255564e-03 7.28492392955404244325734453013865e-01 2.63112829634638112352718053443823e-01 2.72303141744349927466650740370824e-02
            7.28492392955404244325734453013865e-01 2.63112829634638112352718053443823e-01 8.39477740995760515763102205255564e-03 2.72303141744349927466650740370824e-02
            7.28492392955404244325734453013865e-01 8.39477740995760515763102205255564e-03 2.63112829634638112352718053443823e-01 2.72303141744349927466650740370824e-02
            ];
        
    case   8
        
        % ALG. DEG.:    8
        % PTS CARD.:   21
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.4e-13
        
        xyw_bar=[
            4.51890097843999985482987824525480e-02 4.51890097843999985482987824525480e-02 9.09621980431200016781190242909361e-01 2.59935710323064943227944212367220e-02
            4.51890097843999985482987824525480e-02 9.09621980431200016781190242909361e-01 4.51890097843999985482987824525480e-02 2.59935710323064943227944212367220e-02
            9.09621980431200016781190242909361e-01 4.51890097843999985482987824525480e-02 4.51890097843999985482987824525480e-02 2.59935710323064943227944212367220e-02
            4.81519834783300015867979482209194e-01 4.81519834783300015867979482209194e-01 3.69603304333999682640410355816130e-02 5.16172025690128932762235081099789e-02
            4.81519834783300015867979482209194e-01 3.69603304333999682640410355816130e-02 4.81519834783300015867979482209194e-01 5.16172025690128932762235081099789e-02
            3.69603304333999682640410355816130e-02 4.81519834783300015867979482209194e-01 4.81519834783300015867979482209194e-01 5.16172025690128932762235081099789e-02
            4.03603979817900027082799851996242e-01 4.03603979817900027082799851996242e-01 1.92792040364199945834400296007516e-01 9.40800734583734921123721051117172e-02
            4.03603979817900027082799851996242e-01 1.92792040364199945834400296007516e-01 4.03603979817900027082799851996242e-01 9.40800734583734921123721051117172e-02
            1.92792040364199945834400296007516e-01 4.03603979817900027082799851996242e-01 4.03603979817900027082799851996242e-01 9.40800734583734921123721051117172e-02
            7.47512472733899979715488370857202e-01 2.22063165537299989082953288743738e-01 3.04243617288000312015583403990604e-02 3.53517050892088319224981773913896e-02
            7.47512472733899979715488370857202e-01 3.04243617288000312015583403990604e-02 2.22063165537299989082953288743738e-01 3.53517050892088319224981773913896e-02
            2.22063165537299989082953288743738e-01 7.47512472733899979715488370857202e-01 3.04243617288000312015583403990604e-02 3.53517050892088319224981773913896e-02
            2.22063165537299989082953288743738e-01 3.04243617288000312015583403990604e-02 7.47512472733899979715488370857202e-01 3.53517050892088319224981773913896e-02
            3.04243617288000312015583403990604e-02 7.47512472733899979715488370857202e-01 2.22063165537299989082953288743738e-01 3.53517050892088319224981773913896e-02
            3.04243617288000312015583403990604e-02 2.22063165537299989082953288743738e-01 7.47512472733899979715488370857202e-01 3.53517050892088319224981773913896e-02
            1.36991201264899997713797574760974e-01 2.18290070971399990940042812326283e-01 6.44718727763700094612886459799483e-01 4.54695380476113561463158418973762e-02
            1.36991201264899997713797574760974e-01 6.44718727763700094612886459799483e-01 2.18290070971399990940042812326283e-01 4.54695380476113561463158418973762e-02
            2.18290070971399990940042812326283e-01 1.36991201264899997713797574760974e-01 6.44718727763700094612886459799483e-01 4.54695380476113561463158418973762e-02
            2.18290070971399990940042812326283e-01 6.44718727763700094612886459799483e-01 1.36991201264899997713797574760974e-01 4.54695380476113561463158418973762e-02
            6.44718727763700094612886459799483e-01 1.36991201264899997713797574760974e-01 2.18290070971399990940042812326283e-01 4.54695380476113561463158418973762e-02
            6.44718727763700094612886459799483e-01 2.18290070971399990940042812326283e-01 1.36991201264899997713797574760974e-01 4.54695380476113561463158418973762e-02
            ];
        
    case   9
        
        % ALG. DEG.:    9
        % PTS CARD.:   19
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.6e-16
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 9.71357962827988641318910367772332e-02
            4.89682519198737620236272505280795e-01 4.89682519198737620236272505280795e-01 2.06349616025247595274549894384108e-02 3.13347002271390784322591116506374e-02
            4.89682519198737620236272505280795e-01 2.06349616025247595274549894384108e-02 4.89682519198737620236272505280795e-01 3.13347002271390784322591116506374e-02
            2.06349616025247595274549894384108e-02 4.89682519198737620236272505280795e-01 4.89682519198737620236272505280795e-01 3.13347002271390784322591116506374e-02
            4.47295133944527120717360446633393e-02 4.47295133944527120717360446633393e-02 9.10540973211094617489891334116692e-01 2.55776756586980381913409843264162e-02
            4.47295133944527120717360446633393e-02 9.10540973211094617489891334116692e-01 4.47295133944527120717360446633393e-02 2.55776756586980381913409843264162e-02
            9.10540973211094617489891334116692e-01 4.47295133944527120717360446633393e-02 4.47295133944527120717360446633393e-02 2.55776756586980381913409843264162e-02
            4.37089591492936635486188379218220e-01 4.37089591492936635486188379218220e-01 1.25820817014126729027623241563560e-01 7.78275410047742921548774575057905e-02
            4.37089591492936635486188379218220e-01 1.25820817014126729027623241563560e-01 4.37089591492936635486188379218220e-01 7.78275410047742921548774575057905e-02
            1.25820817014126729027623241563560e-01 4.37089591492936635486188379218220e-01 4.37089591492936635486188379218220e-01 7.78275410047742921548774575057905e-02
            1.88203535619032719106158424438036e-01 1.88203535619032719106158424438036e-01 6.23592928761934617298834382381756e-01 7.96477389272102626049942841746088e-02
            1.88203535619032719106158424438036e-01 6.23592928761934617298834382381756e-01 1.88203535619032719106158424438036e-01 7.96477389272102626049942841746088e-02
            6.23592928761934617298834382381756e-01 1.88203535619032719106158424438036e-01 1.88203535619032719106158424438036e-01 7.96477389272102626049942841746088e-02
            7.41198598784498008384957756788936e-01 2.21962989160765705731748198559217e-01 3.68384120547362858832940446518478e-02 4.32835393772893831099146666474553e-02
            7.41198598784498008384957756788936e-01 3.68384120547362858832940446518478e-02 2.21962989160765705731748198559217e-01 4.32835393772893831099146666474553e-02
            2.21962989160765705731748198559217e-01 7.41198598784498008384957756788936e-01 3.68384120547362858832940446518478e-02 4.32835393772893831099146666474553e-02
            2.21962989160765705731748198559217e-01 3.68384120547362858832940446518478e-02 7.41198598784498008384957756788936e-01 4.32835393772893831099146666474553e-02
            3.68384120547362858832940446518478e-02 7.41198598784498008384957756788936e-01 2.21962989160765705731748198559217e-01 4.32835393772893831099146666474553e-02
            3.68384120547362858832940446518478e-02 2.21962989160765705731748198559217e-01 7.41198598784498008384957756788936e-01 4.32835393772893831099146666474553e-02
            ];
        
    case  10
        
        % ALG. DEG.:   10
        % PTS CARD.:   25
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.7e-16
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 8.09374287976228901086983569257427e-02
            4.27273178846775514294620279542869e-01 4.27273178846775514294620279542869e-01 1.45453642306448971410759440914262e-01 7.72985880029631311494142664741958e-02
            4.27273178846775514294620279542869e-01 1.45453642306448971410759440914262e-01 4.27273178846775514294620279542869e-01 7.72985880029631311494142664741958e-02
            1.45453642306448971410759440914262e-01 4.27273178846775514294620279542869e-01 4.27273178846775514294620279542869e-01 7.72985880029631311494142664741958e-02
            1.83099222448675019725428114725219e-01 1.83099222448675019725428114725219e-01 6.33801555102649905037992539291736e-01 7.84576386123717467446780915452109e-02
            1.83099222448675019725428114725219e-01 6.33801555102649905037992539291736e-01 1.83099222448675019725428114725219e-01 7.84576386123717467446780915452109e-02
            6.33801555102649905037992539291736e-01 1.83099222448675019725428114725219e-01 1.83099222448675019725428114725219e-01 7.84576386123717467446780915452109e-02
            4.90434019701130596846638809438446e-01 4.90434019701130596846638809438446e-01 1.91319605977388063067223811231088e-02 1.74691679959294889068477374394206e-02
            4.90434019701130596846638809438446e-01 1.91319605977388063067223811231088e-02 4.90434019701130596846638809438446e-01 1.74691679959294889068477374394206e-02
            1.91319605977388063067223811231088e-02 4.90434019701130596846638809438446e-01 4.90434019701130596846638809438446e-01 1.74691679959294889068477374394206e-02
            1.25724455515805327854295114775596e-02 1.25724455515805327854295114775596e-02 9.74855108896838906673565361415967e-01 4.29237418483282857223359840759258e-03
            1.25724455515805327854295114775596e-02 9.74855108896838906673565361415967e-01 1.25724455515805327854295114775596e-02 4.29237418483282857223359840759258e-03
            9.74855108896838906673565361415967e-01 1.25724455515805327854295114775596e-02 1.25724455515805327854295114775596e-02 4.29237418483282857223359840759258e-03
            6.54268667920066104848331178800436e-01 3.08046001685247705648151850255090e-01 3.76853303946861895035169709444745e-02 3.74688582104676509709761944577622e-02
            6.54268667920066104848331178800436e-01 3.76853303946861895035169709444745e-02 3.08046001685247705648151850255090e-01 3.74688582104676509709761944577622e-02
            3.08046001685247705648151850255090e-01 6.54268667920066104848331178800436e-01 3.76853303946861895035169709444745e-02 3.74688582104676509709761944577622e-02
            3.08046001685247705648151850255090e-01 3.76853303946861895035169709444745e-02 6.54268667920066104848331178800436e-01 3.74688582104676509709761944577622e-02
            3.76853303946861895035169709444745e-02 6.54268667920066104848331178800436e-01 3.08046001685247705648151850255090e-01 3.74688582104676509709761944577622e-02
            3.76853303946861895035169709444745e-02 3.08046001685247705648151850255090e-01 6.54268667920066104848331178800436e-01 3.74688582104676509709761944577622e-02
            1.22804577068559267472025453571405e-01 3.33718337393047875805152102657303e-02 8.43823589192135958825247143977322e-01 2.69493525918799679830240023648003e-02
            1.22804577068559267472025453571405e-01 8.43823589192135958825247143977322e-01 3.33718337393047875805152102657303e-02 2.69493525918799679830240023648003e-02
            3.33718337393047875805152102657303e-02 1.22804577068559267472025453571405e-01 8.43823589192135958825247143977322e-01 2.69493525918799679830240023648003e-02
            3.33718337393047875805152102657303e-02 8.43823589192135958825247143977322e-01 1.22804577068559267472025453571405e-01 2.69493525918799679830240023648003e-02
            8.43823589192135958825247143977322e-01 1.22804577068559267472025453571405e-01 3.33718337393047875805152102657303e-02 2.69493525918799679830240023648003e-02
            8.43823589192135958825247143977322e-01 3.33718337393047875805152102657303e-02 1.22804577068559267472025453571405e-01 2.69493525918799679830240023648003e-02
            ];
        
    case  11
        
        % ALG. DEG.:   11
        % PTS CARD.:   28
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 8.3e-16
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 8.11779602968671493146146644903638e-02
            3.09383552454307840517966354809687e-02 3.09383552454307840517966354809687e-02 9.38123289509138480468664056388661e-01 1.23240435069094898812291916101458e-02
            3.09383552454307840517966354809687e-02 9.38123289509138480468664056388661e-01 3.09383552454307840517966354809687e-02 1.23240435069094898812291916101458e-02
            9.38123289509138480468664056388661e-01 3.09383552454307840517966354809687e-02 3.09383552454307840517966354809687e-02 1.23240435069094898812291916101458e-02
            4.36498181134128859781640130677260e-01 4.36498181134128859781640130677260e-01 1.27003637731742280436719738645479e-01 6.28280097444100932291277672447904e-02
            4.36498181134128859781640130677260e-01 1.27003637731742280436719738645479e-01 4.36498181134128859781640130677260e-01 6.28280097444100932291277672447904e-02
            1.27003637731742280436719738645479e-01 4.36498181134128859781640130677260e-01 4.36498181134128859781640130677260e-01 6.28280097444100932291277672447904e-02
            4.98984763702593259182549445540644e-01 4.98984763702593259182549445540644e-01 2.03047259481348163490110891871154e-03 1.22203790493645263193212002761356e-02
            4.98984763702593259182549445540644e-01 2.03047259481348163490110891871154e-03 4.98984763702593259182549445540644e-01 1.22203790493645263193212002761356e-02
            2.03047259481348163490110891871154e-03 4.98984763702593259182549445540644e-01 4.98984763702593259182549445540644e-01 1.22203790493645263193212002761356e-02
            2.14688197958594340075677564527723e-01 2.14688197958594340075677564527723e-01 5.70623604082811319848644870944554e-01 6.77013489528114908999611998297041e-02
            2.14688197958594340075677564527723e-01 5.70623604082811319848644870944554e-01 2.14688197958594340075677564527723e-01 6.77013489528114908999611998297041e-02
            5.70623604082811319848644870944554e-01 2.14688197958594340075677564527723e-01 2.14688197958594340075677564527723e-01 6.77013489528114908999611998297041e-02
            1.13683104042113389264834211189736e-01 1.13683104042113389264834211189736e-01 7.72633791915773193714755961991614e-01 4.02196936288516868063247500231228e-02
            1.13683104042113389264834211189736e-01 7.72633791915773193714755961991614e-01 1.13683104042113389264834211189736e-01 4.02196936288516868063247500231228e-02
            7.72633791915773193714755961991614e-01 1.13683104042113389264834211189736e-01 1.13683104042113389264834211189736e-01 4.02196936288516868063247500231228e-02
            8.25618766164862849521455245849211e-01 1.59742304591850181605394709549728e-01 1.46389292432869688731500446010614e-02 1.47622727177160982525716548252603e-02
            8.25618766164862849521455245849211e-01 1.46389292432869688731500446010614e-02 1.59742304591850181605394709549728e-01 1.47622727177160982525716548252603e-02
            1.59742304591850181605394709549728e-01 8.25618766164862849521455245849211e-01 1.46389292432869688731500446010614e-02 1.47622727177160982525716548252603e-02
            1.59742304591850181605394709549728e-01 1.46389292432869688731500446010614e-02 8.25618766164862849521455245849211e-01 1.47622727177160982525716548252603e-02
            1.46389292432869688731500446010614e-02 8.25618766164862849521455245849211e-01 1.59742304591850181605394709549728e-01 1.47622727177160982525716548252603e-02
            1.46389292432869688731500446010614e-02 1.59742304591850181605394709549728e-01 8.25618766164862849521455245849211e-01 1.47622727177160982525716548252603e-02
            6.40472310134865274733328988077119e-01 3.11783715709599007492869304769556e-01 4.77439741555357177738017071533250e-02 4.07279964582990330890233110494592e-02
            6.40472310134865274733328988077119e-01 4.77439741555357177738017071533250e-02 3.11783715709599007492869304769556e-01 4.07279964582990330890233110494592e-02
            3.11783715709599007492869304769556e-01 6.40472310134865274733328988077119e-01 4.77439741555357177738017071533250e-02 4.07279964582990330890233110494592e-02
            3.11783715709599007492869304769556e-01 4.77439741555357177738017071533250e-02 6.40472310134865274733328988077119e-01 4.07279964582990330890233110494592e-02
            4.77439741555357177738017071533250e-02 6.40472310134865274733328988077119e-01 3.11783715709599007492869304769556e-01 4.07279964582990330890233110494592e-02
            4.77439741555357177738017071533250e-02 3.11783715709599007492869304769556e-01 6.40472310134865274733328988077119e-01 4.07279964582990330890233110494592e-02
            ];
        
    case  12
        
        % ALG. DEG.:   12
        % PTS CARD.:   33
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.8e-16
        
        xyw_bar=[
            2.13173504532103708086765436746646e-02 2.13173504532103708086765436746646e-02 9.57365299093579258382646912650671e-01 6.16626105155902021204283869337814e-03
            2.13173504532103708086765436746646e-02 9.57365299093579258382646912650671e-01 2.13173504532103708086765436746646e-02 6.16626105155902021204283869337814e-03
            9.57365299093579258382646912650671e-01 2.13173504532103708086765436746646e-02 2.13173504532103708086765436746646e-02 6.16626105155902021204283869337814e-03
            2.71210385012115917646724483347498e-01 2.71210385012115917646724483347498e-01 4.57579229975768164706551033305004e-01 6.28582242178851308134213127232215e-02
            2.71210385012115917646724483347498e-01 4.57579229975768164706551033305004e-01 2.71210385012115917646724483347498e-01 6.28582242178851308134213127232215e-02
            4.57579229975768164706551033305004e-01 2.71210385012115917646724483347498e-01 2.71210385012115917646724483347498e-01 6.28582242178851308134213127232215e-02
            1.27576145541585916909355091775069e-01 1.27576145541585916909355091775069e-01 7.44847708916828166181289816449862e-01 3.47961129307089583884149419645837e-02
            1.27576145541585916909355091775069e-01 7.44847708916828166181289816449862e-01 1.27576145541585916909355091775069e-01 3.47961129307089583884149419645837e-02
            7.44847708916828166181289816449862e-01 1.27576145541585916909355091775069e-01 1.27576145541585916909355091775069e-01 3.47961129307089583884149419645837e-02
            4.39724392294460253438614927290473e-01 4.39724392294460253438614927290473e-01 1.20551215411079493122770145419054e-01 4.36925445380384261384598687527614e-02
            4.39724392294460253438614927290473e-01 1.20551215411079493122770145419054e-01 4.39724392294460253438614927290473e-01 4.36925445380384261384598687527614e-02
            1.20551215411079493122770145419054e-01 4.39724392294460253438614927290473e-01 4.39724392294460253438614927290473e-01 4.36925445380384261384598687527614e-02
            4.88217389773804855135352909201174e-01 4.88217389773804855135352909201174e-01 2.35652204523902897292941815976519e-02 2.57310664404553461614977294402706e-02
            4.88217389773804855135352909201174e-01 2.35652204523902897292941815976519e-02 4.88217389773804855135352909201174e-01 2.57310664404553461614977294402706e-02
            2.35652204523902897292941815976519e-02 4.88217389773804855135352909201174e-01 4.88217389773804855135352909201174e-01 2.57310664404553461614977294402706e-02
            6.95836086787803398401308641041396e-01 2.81325580989939538767430349253118e-01 2.28383322222570628312610097054858e-02 2.23567732023034551613882570109126e-02
            6.95836086787803398401308641041396e-01 2.28383322222570628312610097054858e-02 2.81325580989939538767430349253118e-01 2.23567732023034551613882570109126e-02
            2.81325580989939538767430349253118e-01 6.95836086787803398401308641041396e-01 2.28383322222570628312610097054858e-02 2.23567732023034551613882570109126e-02
            2.81325580989939538767430349253118e-01 2.28383322222570628312610097054858e-02 6.95836086787803398401308641041396e-01 2.23567732023034551613882570109126e-02
            2.28383322222570628312610097054858e-02 6.95836086787803398401308641041396e-01 2.81325580989939538767430349253118e-01 2.23567732023034551613882570109126e-02
            2.28383322222570628312610097054858e-02 2.81325580989939538767430349253118e-01 6.95836086787803398401308641041396e-01 2.23567732023034551613882570109126e-02
            8.58014033544072685089076912845485e-01 1.16251915907597147548102611835930e-01 2.57340505483301673628204753185855e-02 1.73162311086588992237178530331221e-02
            8.58014033544072685089076912845485e-01 2.57340505483301673628204753185855e-02 1.16251915907597147548102611835930e-01 1.73162311086588992237178530331221e-02
            1.16251915907597147548102611835930e-01 8.58014033544072685089076912845485e-01 2.57340505483301673628204753185855e-02 1.73162311086588992237178530331221e-02
            1.16251915907597147548102611835930e-01 2.57340505483301673628204753185855e-02 8.58014033544072685089076912845485e-01 1.73162311086588992237178530331221e-02
            2.57340505483301673628204753185855e-02 8.58014033544072685089076912845485e-01 1.16251915907597147548102611835930e-01 1.73162311086588992237178530331221e-02
            2.57340505483301673628204753185855e-02 1.16251915907597147548102611835930e-01 8.58014033544072685089076912845485e-01 1.73162311086588992237178530331221e-02
            6.08943235779787817030239693849580e-01 2.75713269685514184015318051024224e-01 1.15343494534697998954442255126196e-01 4.03715577663809471742339951561007e-02
            6.08943235779787817030239693849580e-01 1.15343494534697998954442255126196e-01 2.75713269685514184015318051024224e-01 4.03715577663809471742339951561007e-02
            2.75713269685514184015318051024224e-01 6.08943235779787817030239693849580e-01 1.15343494534697998954442255126196e-01 4.03715577663809471742339951561007e-02
            2.75713269685514184015318051024224e-01 1.15343494534697998954442255126196e-01 6.08943235779787817030239693849580e-01 4.03715577663809471742339951561007e-02
            1.15343494534697998954442255126196e-01 6.08943235779787817030239693849580e-01 2.75713269685514184015318051024224e-01 4.03715577663809471742339951561007e-02
            1.15343494534697998954442255126196e-01 2.75713269685514184015318051024224e-01 6.08943235779787817030239693849580e-01 4.03715577663809471742339951561007e-02
            ];
        
    case  13
        
        % ALG. DEG.:   13
        % PTS CARD.:   37
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.8e-16
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 6.79600365868316680373339977450087e-02
            4.26941414259800422481561099630198e-01 4.26941414259800422481561099630198e-01 1.46117171480399155036877800739603e-01 5.56019675304533569915044211029453e-02
            4.26941414259800422481561099630198e-01 1.46117171480399155036877800739603e-01 4.26941414259800422481561099630198e-01 5.56019675304533569915044211029453e-02
            1.46117171480399155036877800739603e-01 4.26941414259800422481561099630198e-01 4.26941414259800422481561099630198e-01 5.56019675304533569915044211029453e-02
            2.21372286291832892635156326832657e-01 2.21372286291832892635156326832657e-01 5.57255427416334159218536115076859e-01 5.82784851192000089015188279972790e-02
            2.21372286291832892635156326832657e-01 5.57255427416334159218536115076859e-01 2.21372286291832892635156326832657e-01 5.82784851192000089015188279972790e-02
            5.57255427416334159218536115076859e-01 2.21372286291832892635156326832657e-01 2.21372286291832892635156326832657e-01 5.82784851192000089015188279972790e-02
            2.15096811088431837111212274749050e-02 2.15096811088431837111212274749050e-02 9.56980637782313681150014872400789e-01 6.05233710353917427754444346987839e-03
            2.15096811088431837111212274749050e-02 9.56980637782313681150014872400789e-01 2.15096811088431837111212274749050e-02 6.05233710353917427754444346987839e-03
            9.56980637782313681150014872400789e-01 2.15096811088431837111212274749050e-02 2.15096811088431837111212274749050e-02 6.05233710353917427754444346987839e-03
            4.89076946452539351728461269885884e-01 4.89076946452539351728461269885884e-01 2.18461070949212965430774602282327e-02 2.39944019288947409107581876241966e-02
            4.89076946452539351728461269885884e-01 2.18461070949212965430774602282327e-02 4.89076946452539351728461269885884e-01 2.39944019288947409107581876241966e-02
            2.18461070949212965430774602282327e-02 4.89076946452539351728461269885884e-01 4.89076946452539351728461269885884e-01 2.39944019288947409107581876241966e-02
            6.23545995553675624911704744590679e-01 3.08441760892117777803633771327441e-01 6.80122435542065972846614840818802e-02 3.46412761408483871328201075812103e-02
            6.23545995553675624911704744590679e-01 6.80122435542065972846614840818802e-02 3.08441760892117777803633771327441e-01 3.46412761408483871328201075812103e-02
            3.08441760892117777803633771327441e-01 6.23545995553675624911704744590679e-01 6.80122435542065972846614840818802e-02 3.46412761408483871328201075812103e-02
            3.08441760892117777803633771327441e-01 6.80122435542065972846614840818802e-02 6.23545995553675624911704744590679e-01 3.46412761408483871328201075812103e-02
            6.80122435542065972846614840818802e-02 6.23545995553675624911704744590679e-01 3.08441760892117777803633771327441e-01 3.46412761408483871328201075812103e-02
            6.80122435542065972846614840818802e-02 3.08441760892117777803633771327441e-01 6.23545995553675624911704744590679e-01 3.46412761408483871328201075812103e-02
            8.64707770295442768038185477053048e-01 1.10922042803463391513751901129581e-01 2.43701869010938404480626218173711e-02 1.49654011051656744835103296509260e-02
            8.64707770295442768038185477053048e-01 2.43701869010938404480626218173711e-02 1.10922042803463391513751901129581e-01 1.49654011051656744835103296509260e-02
            1.10922042803463391513751901129581e-01 8.64707770295442768038185477053048e-01 2.43701869010938404480626218173711e-02 1.49654011051656744835103296509260e-02
            1.10922042803463391513751901129581e-01 2.43701869010938404480626218173711e-02 8.64707770295442768038185477053048e-01 1.49654011051656744835103296509260e-02
            2.43701869010938404480626218173711e-02 8.64707770295442768038185477053048e-01 1.10922042803463391513751901129581e-01 1.49654011051656744835103296509260e-02
            2.43701869010938404480626218173711e-02 1.10922042803463391513751901129581e-01 8.64707770295442768038185477053048e-01 1.49654011051656744835103296509260e-02
            7.48507115899952224502555964136263e-01 1.63597401067850478639797984214965e-01 8.78954830321972968576460516487714e-02 2.41790398115938297285421043625320e-02
            7.48507115899952224502555964136263e-01 8.78954830321972968576460516487714e-02 1.63597401067850478639797984214965e-01 2.41790398115938297285421043625320e-02
            1.63597401067850478639797984214965e-01 7.48507115899952224502555964136263e-01 8.78954830321972968576460516487714e-02 2.41790398115938297285421043625320e-02
            1.63597401067850478639797984214965e-01 8.78954830321972968576460516487714e-02 7.48507115899952224502555964136263e-01 2.41790398115938297285421043625320e-02
            8.78954830321972968576460516487714e-02 7.48507115899952224502555964136263e-01 1.63597401067850478639797984214965e-01 2.41790398115938297285421043625320e-02
            8.78954830321972968576460516487714e-02 1.63597401067850478639797984214965e-01 7.48507115899952224502555964136263e-01 2.41790398115938297285421043625320e-02
            7.22357793124188019007192451681476e-01 2.72515817773429647186134161529480e-01 5.12638910238233380667338678904343e-03 9.59068100354326658063541799492668e-03
            7.22357793124188019007192451681476e-01 5.12638910238233380667338678904343e-03 2.72515817773429647186134161529480e-01 9.59068100354326658063541799492668e-03
            2.72515817773429647186134161529480e-01 7.22357793124188019007192451681476e-01 5.12638910238233380667338678904343e-03 9.59068100354326658063541799492668e-03
            2.72515817773429647186134161529480e-01 5.12638910238233380667338678904343e-03 7.22357793124188019007192451681476e-01 9.59068100354326658063541799492668e-03
            5.12638910238233380667338678904343e-03 7.22357793124188019007192451681476e-01 2.72515817773429647186134161529480e-01 9.59068100354326658063541799492668e-03
            5.12638910238233380667338678904343e-03 2.72515817773429647186134161529480e-01 7.22357793124188019007192451681476e-01 9.59068100354326658063541799492668e-03
            ];
        
    case  14
        
        % ALG. DEG.:   14
        % PTS CARD.:   42
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.6e-16
        
        xyw_bar=[
            1.77205532412543442788432912493590e-01 1.77205532412543442788432912493590e-01 6.45588935174913114423134175012819e-01 4.21625887369930230086367828334915e-02
            1.77205532412543442788432912493590e-01 6.45588935174913114423134175012819e-01 1.77205532412543442788432912493590e-01 4.21625887369930230086367828334915e-02
            6.45588935174913114423134175012819e-01 1.77205532412543442788432912493590e-01 1.77205532412543442788432912493590e-01 4.21625887369930230086367828334915e-02
            1.93909612487010475645465845673243e-02 1.93909612487010475645465845673243e-02 9.61218077502597911809800734772580e-01 4.92340360240008274861356341034480e-03
            1.93909612487010475645465845673243e-02 9.61218077502597911809800734772580e-01 1.93909612487010475645465845673243e-02 4.92340360240008274861356341034480e-03
            9.61218077502597911809800734772580e-01 1.93909612487010475645465845673243e-02 1.93909612487010475645465845673243e-02 4.92340360240008274861356341034480e-03
            6.17998830908726010102682835167798e-02 6.17998830908726010102682835167798e-02 8.76400233818254825735039048595354e-01 1.44336996697766711122978833259367e-02
            6.17998830908726010102682835167798e-02 8.76400233818254825735039048595354e-01 6.17998830908726010102682835167798e-02 1.44336996697766711122978833259367e-02
            8.76400233818254825735039048595354e-01 6.17998830908726010102682835167798e-02 6.17998830908726010102682835167798e-02 1.44336996697766711122978833259367e-02
            4.17644719340453940414903399869218e-01 4.17644719340453940414903399869218e-01 1.64710561319092119170193200261565e-01 3.27883535441253551323548265372665e-02
            4.17644719340453940414903399869218e-01 1.64710561319092119170193200261565e-01 4.17644719340453940414903399869218e-01 3.27883535441253551323548265372665e-02
            1.64710561319092119170193200261565e-01 4.17644719340453940414903399869218e-01 4.17644719340453940414903399869218e-01 3.27883535441253551323548265372665e-02
            4.88963910362178622026618768359185e-01 4.88963910362178622026618768359185e-01 2.20721792756427559467624632816296e-02 2.18835813694288928987763398481547e-02
            4.88963910362178622026618768359185e-01 2.20721792756427559467624632816296e-02 4.88963910362178622026618768359185e-01 2.18835813694288928987763398481547e-02
            2.20721792756427559467624632816296e-02 4.88963910362178622026618768359185e-01 4.88963910362178622026618768359185e-01 2.18835813694288928987763398481547e-02
            2.73477528308838646609046918456443e-01 2.73477528308838646609046918456443e-01 4.53044943382322706781906163087115e-01 5.17741045072915989644357637189387e-02
            2.73477528308838646609046918456443e-01 4.53044943382322706781906163087115e-01 2.73477528308838646609046918456443e-01 5.17741045072915989644357637189387e-02
            4.53044943382322706781906163087115e-01 2.73477528308838646609046918456443e-01 2.73477528308838646609046918456443e-01 5.17741045072915989644357637189387e-02
            1.72266687821355568566161764465505e-01 5.71247574036479396908383421305189e-02 7.70608554774996457048530373867834e-01 2.46657532125636805298007914188929e-02
            1.72266687821355568566161764465505e-01 7.70608554774996457048530373867834e-01 5.71247574036479396908383421305189e-02 2.46657532125636805298007914188929e-02
            5.71247574036479396908383421305189e-02 1.72266687821355568566161764465505e-01 7.70608554774996457048530373867834e-01 2.46657532125636805298007914188929e-02
            5.71247574036479396908383421305189e-02 7.70608554774996457048530373867834e-01 1.72266687821355568566161764465505e-01 2.46657532125636805298007914188929e-02
            7.70608554774996457048530373867834e-01 1.72266687821355568566161764465505e-01 5.71247574036479396908383421305189e-02 2.46657532125636805298007914188929e-02
            7.70608554774996457048530373867834e-01 5.71247574036479396908383421305189e-02 1.72266687821355568566161764465505e-01 2.46657532125636805298007914188929e-02
            5.70222290846683188547672216373030e-01 9.29162493569718195285389583659708e-02 3.36861459796344964168213209632086e-01 3.85715107870606907503052696029044e-02
            5.70222290846683188547672216373030e-01 3.36861459796344964168213209632086e-01 9.29162493569718195285389583659708e-02 3.85715107870606907503052696029044e-02
            9.29162493569718195285389583659708e-02 5.70222290846683188547672216373030e-01 3.36861459796344964168213209632086e-01 3.85715107870606907503052696029044e-02
            9.29162493569718195285389583659708e-02 3.36861459796344964168213209632086e-01 5.70222290846683188547672216373030e-01 3.85715107870606907503052696029044e-02
            3.36861459796344964168213209632086e-01 5.70222290846683188547672216373030e-01 9.29162493569718195285389583659708e-02 3.85715107870606907503052696029044e-02
            3.36861459796344964168213209632086e-01 9.29162493569718195285389583659708e-02 5.70222290846683188547672216373030e-01 3.85715107870606907503052696029044e-02
            2.98372882136257733254325330563006e-01 1.46469500556544104941103512373957e-02 6.86980167808087904823821645550197e-01 1.44363081135338439109361274859111e-02
            2.98372882136257733254325330563006e-01 6.86980167808087904823821645550197e-01 1.46469500556544104941103512373957e-02 1.44363081135338439109361274859111e-02
            1.46469500556544104941103512373957e-02 2.98372882136257733254325330563006e-01 6.86980167808087904823821645550197e-01 1.44363081135338439109361274859111e-02
            1.46469500556544104941103512373957e-02 6.86980167808087904823821645550197e-01 2.98372882136257733254325330563006e-01 1.44363081135338439109361274859111e-02
            6.86980167808087904823821645550197e-01 2.98372882136257733254325330563006e-01 1.46469500556544104941103512373957e-02 1.44363081135338439109361274859111e-02
            6.86980167808087904823821645550197e-01 1.46469500556544104941103512373957e-02 2.98372882136257733254325330563006e-01 1.44363081135338439109361274859111e-02
            1.18974497696956851844340974366787e-01 1.26833093287202509219513402172197e-03 8.79757171370171064950227446388453e-01 5.01022883850067279193352121069438e-03
            1.18974497696956851844340974366787e-01 8.79757171370171064950227446388453e-01 1.26833093287202509219513402172197e-03 5.01022883850067279193352121069438e-03
            1.26833093287202509219513402172197e-03 1.18974497696956851844340974366787e-01 8.79757171370171064950227446388453e-01 5.01022883850067279193352121069438e-03
            1.26833093287202509219513402172197e-03 8.79757171370171064950227446388453e-01 1.18974497696956851844340974366787e-01 5.01022883850067279193352121069438e-03
            8.79757171370171064950227446388453e-01 1.18974497696956851844340974366787e-01 1.26833093287202509219513402172197e-03 5.01022883850067279193352121069438e-03
            8.79757171370171064950227446388453e-01 1.26833093287202509219513402172197e-03 1.18974497696956851844340974366787e-01 5.01022883850067279193352121069438e-03
            ];
        
    case  15
        
        % ALG. DEG.:   15
        % PTS CARD.:   49
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.8e-16
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 2.35712670319063594126518523808045e-02
            1.10222296228346866575620310868544e-01 1.10222296228346866575620310868544e-01 7.79555407543306211337608147005085e-01 1.51731495572117146275470389582551e-02
            1.10222296228346866575620310868544e-01 7.79555407543306211337608147005085e-01 1.10222296228346866575620310868544e-01 1.51731495572117146275470389582551e-02
            7.79555407543306211337608147005085e-01 1.10222296228346866575620310868544e-01 1.10222296228346866575620310868544e-01 1.51731495572117146275470389582551e-02
            5.19764330100343507901072825916344e-02 5.19764330100343507901072825916344e-02 8.96047133979931298419785434816731e-01 1.29760012839288500774426182715615e-02
            5.19764330100343507901072825916344e-02 8.96047133979931298419785434816731e-01 5.19764330100343507901072825916344e-02 1.29760012839288500774426182715615e-02
            8.96047133979931298419785434816731e-01 5.19764330100343507901072825916344e-02 5.19764330100343507901072825916344e-02 1.29760012839288500774426182715615e-02
            4.91145658075325564073665418618475e-01 4.91145658075325564073665418618475e-01 1.77086838493488718526691627630498e-02 1.70662959680061669665018797559242e-02
            4.91145658075325564073665418618475e-01 1.77086838493488718526691627630498e-02 4.91145658075325564073665418618475e-01 1.70662959680061669665018797559242e-02
            1.77086838493488718526691627630498e-02 4.91145658075325564073665418618475e-01 4.91145658075325564073665418618475e-01 1.70662959680061669665018797559242e-02
            3.93157188884358854696898788461112e-01 3.93157188884358854696898788461112e-01 2.13685622231282290606202423077775e-01 4.57600194627376319345124500159727e-02
            3.93157188884358854696898788461112e-01 2.13685622231282290606202423077775e-01 3.93157188884358854696898788461112e-01 4.57600194627376319345124500159727e-02
            2.13685622231282290606202423077775e-01 3.93157188884358854696898788461112e-01 3.93157188884358854696898788461112e-01 4.57600194627376319345124500159727e-02
            3.73744048757291905116240116058179e-02 9.62518352230012141923509716434637e-01 1.07242894258674503760175866773352e-04 2.22757447282223271581291790255364e-03
            3.73744048757291905116240116058179e-02 1.07242894258674503760175866773352e-04 9.62518352230012141923509716434637e-01 2.22757447282223271581291790255364e-03
            9.62518352230012141923509716434637e-01 3.73744048757291905116240116058179e-02 1.07242894258674503760175866773352e-04 2.22757447282223271581291790255364e-03
            9.62518352230012141923509716434637e-01 1.07242894258674503760175866773352e-04 3.73744048757291905116240116058179e-02 2.22757447282223271581291790255364e-03
            1.07242894258674503760175866773352e-04 3.73744048757291905116240116058179e-02 9.62518352230012141923509716434637e-01 2.22757447282223271581291790255364e-03
            1.07242894258674503760175866773352e-04 9.62518352230012141923509716434637e-01 3.73744048757291905116240116058179e-02 2.22757447282223271581291790255364e-03
            2.48248777984673207885890633406234e-01 1.93166698545214177151763124129502e-01 5.58584523470112670473497473722091e-01 2.70101416598694868409502589656768e-02
            2.48248777984673207885890633406234e-01 5.58584523470112670473497473722091e-01 1.93166698545214177151763124129502e-01 2.70101416598694868409502589656768e-02
            1.93166698545214177151763124129502e-01 2.48248777984673207885890633406234e-01 5.58584523470112670473497473722091e-01 2.70101416598694868409502589656768e-02
            1.93166698545214177151763124129502e-01 5.58584523470112670473497473722091e-01 2.48248777984673207885890633406234e-01 2.70101416598694868409502589656768e-02
            5.58584523470112670473497473722091e-01 2.48248777984673207885890633406234e-01 1.93166698545214177151763124129502e-01 2.70101416598694868409502589656768e-02
            5.58584523470112670473497473722091e-01 1.93166698545214177151763124129502e-01 2.48248777984673207885890633406234e-01 2.70101416598694868409502589656768e-02
            2.06994022748302169967260510929918e-01 8.68959088354996217473669162245642e-02 7.06110068416198277674311611917801e-01 2.60837796395875821942045291734757e-02
            2.06994022748302169967260510929918e-01 7.06110068416198277674311611917801e-01 8.68959088354996217473669162245642e-02 2.60837796395875821942045291734757e-02
            8.68959088354996217473669162245642e-02 2.06994022748302169967260510929918e-01 7.06110068416198277674311611917801e-01 2.60837796395875821942045291734757e-02
            8.68959088354996217473669162245642e-02 7.06110068416198277674311611917801e-01 2.06994022748302169967260510929918e-01 2.60837796395875821942045291734757e-02
            7.06110068416198277674311611917801e-01 2.06994022748302169967260510929918e-01 8.68959088354996217473669162245642e-02 2.60837796395875821942045291734757e-02
            7.06110068416198277674311611917801e-01 8.68959088354996217473669162245642e-02 2.06994022748302169967260510929918e-01 2.60837796395875821942045291734757e-02
            1.48541105269547085709902489725209e-01 1.74368253984543072898638627066248e-02 8.34022069331998627816915359289851e-01 1.21101532770282929035721153354643e-02
            1.48541105269547085709902489725209e-01 8.34022069331998627816915359289851e-01 1.74368253984543072898638627066248e-02 1.21101532770282929035721153354643e-02
            1.74368253984543072898638627066248e-02 1.48541105269547085709902489725209e-01 8.34022069331998627816915359289851e-01 1.21101532770282929035721153354643e-02
            1.74368253984543072898638627066248e-02 8.34022069331998627816915359289851e-01 1.48541105269547085709902489725209e-01 1.21101532770282929035721153354643e-02
            8.34022069331998627816915359289851e-01 1.48541105269547085709902489725209e-01 1.74368253984543072898638627066248e-02 1.21101532770282929035721153354643e-02
            8.34022069331998627816915359289851e-01 1.74368253984543072898638627066248e-02 1.48541105269547085709902489725209e-01 1.21101532770282929035721153354643e-02
            3.06742379235963813499665775452740e-01 1.74925109582576600308900793834255e-02 6.75765109805778529938891097117448e-01 1.56478505968044544205408641346366e-02
            3.06742379235963813499665775452740e-01 6.75765109805778529938891097117448e-01 1.74925109582576600308900793834255e-02 1.56478505968044544205408641346366e-02
            1.74925109582576600308900793834255e-02 3.06742379235963813499665775452740e-01 6.75765109805778529938891097117448e-01 1.56478505968044544205408641346366e-02
            1.74925109582576600308900793834255e-02 6.75765109805778529938891097117448e-01 3.06742379235963813499665775452740e-01 1.56478505968044544205408641346366e-02
            6.75765109805778529938891097117448e-01 3.06742379235963813499665775452740e-01 1.74925109582576600308900793834255e-02 1.56478505968044544205408641346366e-02
            6.75765109805778529938891097117448e-01 1.74925109582576600308900793834255e-02 3.06742379235963813499665775452740e-01 1.56478505968044544205408641346366e-02
            3.67031987542204729546568842124543e-01 9.03480217586455636169517902089865e-02 5.42619990699149679080903752037557e-01 3.41708893792948137368981065264961e-02
            3.67031987542204729546568842124543e-01 5.42619990699149679080903752037557e-01 9.03480217586455636169517902089865e-02 3.41708893792948137368981065264961e-02
            9.03480217586455636169517902089865e-02 3.67031987542204729546568842124543e-01 5.42619990699149679080903752037557e-01 3.41708893792948137368981065264961e-02
            9.03480217586455636169517902089865e-02 5.42619990699149679080903752037557e-01 3.67031987542204729546568842124543e-01 3.41708893792948137368981065264961e-02
            5.42619990699149679080903752037557e-01 3.67031987542204729546568842124543e-01 9.03480217586455636169517902089865e-02 3.41708893792948137368981065264961e-02
            5.42619990699149679080903752037557e-01 9.03480217586455636169517902089865e-02 3.67031987542204729546568842124543e-01 3.41708893792948137368981065264961e-02
            ];
        
    case  16
        
        % ALG. DEG.:   16
        % PTS CARD.:   55
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 8.1e-16
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 4.80221886803771230711745943153801e-02
            8.17949831313738717764039165558643e-02 8.17949831313738717764039165558643e-02 8.36410033737252200936040935630444e-01 1.47091003068019372801433775066471e-02
            8.17949831313738717764039165558643e-02 8.36410033737252200936040935630444e-01 8.17949831313738717764039165558643e-02 1.47091003068019372801433775066471e-02
            8.36410033737252200936040935630444e-01 8.17949831313738717764039165558643e-02 8.17949831313738717764039165558643e-02 1.47091003068019372801433775066471e-02
            1.65300601969779636934987365748384e-01 1.65300601969779636934987365748384e-01 6.69398796060440726130025268503232e-01 2.95445865493192752482265461821953e-02
            1.65300601969779636934987365748384e-01 6.69398796060440726130025268503232e-01 1.65300601969779636934987365748384e-01 2.95445865493192752482265461821953e-02
            6.69398796060440726130025268503232e-01 1.65300601969779636934987365748384e-01 1.65300601969779636934987365748384e-01 2.95445865493192752482265461821953e-02
            4.68592105349461385266351953760022e-01 4.68592105349461385266351953760022e-01 6.28157893010772294672960924799554e-02 2.61250173510883948457816927657404e-02
            4.68592105349461385266351953760022e-01 6.28157893010772294672960924799554e-02 4.68592105349461385266351953760022e-01 2.61250173510883948457816927657404e-02
            6.28157893010772294672960924799554e-02 4.68592105349461385266351953760022e-01 4.68592105349461385266351953760022e-01 2.61250173510883948457816927657404e-02
            1.44388134454166826986520888453924e-02 1.44388134454166826986520888453924e-02 9.71122373109166603377673254726687e-01 2.78038735239000871571346529265156e-03
            1.44388134454166826986520888453924e-02 9.71122373109166603377673254726687e-01 1.44388134454166826986520888453924e-02 2.78038735239000871571346529265156e-03
            9.71122373109166603377673254726687e-01 1.44388134454166826986520888453924e-02 1.44388134454166826986520888453924e-02 2.78038735239000871571346529265156e-03
            2.41784285391783343444060960791830e-01 2.41784285391783343444060960791830e-01 5.16431429216433368623029309674166e-01 3.18217730005366725154480889159458e-02
            2.41784285391783343444060960791830e-01 5.16431429216433368623029309674166e-01 2.41784285391783343444060960791830e-01 3.18217730005366725154480889159458e-02
            5.16431429216433368623029309674166e-01 2.41784285391783343444060960791830e-01 2.41784285391783343444060960791830e-01 3.18217730005366725154480889159458e-02
            4.95310342987769991474067410308635e-01 4.95310342987769991474067410308635e-01 9.37931402446001705186517938273028e-03 8.64583434950966581156173162980849e-03
            4.95310342987769991474067410308635e-01 9.37931402446001705186517938273028e-03 4.95310342987769991474067410308635e-01 8.64583434950966581156173162980849e-03
            9.37931402446001705186517938273028e-03 4.95310342987769991474067410308635e-01 4.95310342987769991474067410308635e-01 8.64583434950966581156173162980849e-03
            6.50513402661352269085170973994536e-01 3.31399744537089535612039981060661e-01 1.80868528015581953027890449448023e-02 1.43003329044953733650835658863798e-02
            6.50513402661352269085170973994536e-01 1.80868528015581953027890449448023e-02 3.31399744537089535612039981060661e-01 1.43003329044953733650835658863798e-02
            3.31399744537089535612039981060661e-01 6.50513402661352269085170973994536e-01 1.80868528015581953027890449448023e-02 1.43003329044953733650835658863798e-02
            3.31399744537089535612039981060661e-01 1.80868528015581953027890449448023e-02 6.50513402661352269085170973994536e-01 1.43003329044953733650835658863798e-02
            1.80868528015581953027890449448023e-02 6.50513402661352269085170973994536e-01 3.31399744537089535612039981060661e-01 1.43003329044953733650835658863798e-02
            1.80868528015581953027890449448023e-02 3.31399744537089535612039981060661e-01 6.50513402661352269085170973994536e-01 1.43003329044953733650835658863798e-02
            6.04011281495997032209288590820506e-01 3.03247162749942178461282082935213e-01 9.27415557540607893294293262442807e-02 2.78497772036008471585599721720428e-02
            6.04011281495997032209288590820506e-01 9.27415557540607893294293262442807e-02 3.03247162749942178461282082935213e-01 2.78497772036008471585599721720428e-02
            3.03247162749942178461282082935213e-01 6.04011281495997032209288590820506e-01 9.27415557540607893294293262442807e-02 2.78497772036008471585599721720428e-02
            3.03247162749942178461282082935213e-01 9.27415557540607893294293262442807e-02 6.04011281495997032209288590820506e-01 2.78497772036008471585599721720428e-02
            9.27415557540607893294293262442807e-02 6.04011281495997032209288590820506e-01 3.03247162749942178461282082935213e-01 2.78497772036008471585599721720428e-02
            9.27415557540607893294293262442807e-02 3.03247162749942178461282082935213e-01 6.04011281495997032209288590820506e-01 2.78497772036008471585599721720428e-02
            8.02168257574741638826765210978920e-01 1.88028059521237161932916137629945e-01 9.80368290402119924031865139113506e-03 7.04167340663610155648832744645915e-03
            8.02168257574741638826765210978920e-01 9.80368290402119924031865139113506e-03 1.88028059521237161932916137629945e-01 7.04167340663610155648832744645915e-03
            1.88028059521237161932916137629945e-01 8.02168257574741638826765210978920e-01 9.80368290402119924031865139113506e-03 7.04167340663610155648832744645915e-03
            1.88028059521237161932916137629945e-01 9.80368290402119924031865139113506e-03 8.02168257574741638826765210978920e-01 7.04167340663610155648832744645915e-03
            9.80368290402119924031865139113506e-03 8.02168257574741638826765210978920e-01 1.88028059521237161932916137629945e-01 7.04167340663610155648832744645915e-03
            9.80368290402119924031865139113506e-03 1.88028059521237161932916137629945e-01 8.02168257574741638826765210978920e-01 7.04167340663610155648832744645915e-03
            7.56505606442828382185439295426477e-01 1.83504668522296854327180426480481e-01 5.99897250348747634873802780930419e-02 1.78998382599337389764482253440292e-02
            7.56505606442828382185439295426477e-01 5.99897250348747634873802780930419e-02 1.83504668522296854327180426480481e-01 1.78998382599337389764482253440292e-02
            1.83504668522296854327180426480481e-01 7.56505606442828382185439295426477e-01 5.99897250348747634873802780930419e-02 1.78998382599337389764482253440292e-02
            1.83504668522296854327180426480481e-01 5.99897250348747634873802780930419e-02 7.56505606442828382185439295426477e-01 1.78998382599337389764482253440292e-02
            5.99897250348747634873802780930419e-02 7.56505606442828382185439295426477e-01 1.83504668522296854327180426480481e-01 1.78998382599337389764482253440292e-02
            5.99897250348747634873802780930419e-02 1.83504668522296854327180426480481e-01 7.56505606442828382185439295426477e-01 1.78998382599337389764482253440292e-02
            4.65938438714118163819222218080540e-01 3.59645948797504610983111206223839e-01 1.74415612488377280708817806953448e-01 2.74582003843497797490602607695109e-02
            4.65938438714118163819222218080540e-01 1.74415612488377280708817806953448e-01 3.59645948797504610983111206223839e-01 2.74582003843497797490602607695109e-02
            3.59645948797504610983111206223839e-01 4.65938438714118163819222218080540e-01 1.74415612488377280708817806953448e-01 2.74582003843497797490602607695109e-02
            3.59645948797504610983111206223839e-01 1.74415612488377280708817806953448e-01 4.65938438714118163819222218080540e-01 2.74582003843497797490602607695109e-02
            1.74415612488377280708817806953448e-01 4.65938438714118163819222218080540e-01 3.59645948797504610983111206223839e-01 2.74582003843497797490602607695109e-02
            1.74415612488377280708817806953448e-01 3.59645948797504610983111206223839e-01 4.65938438714118163819222218080540e-01 2.74582003843497797490602607695109e-02
            9.06394843992041510105650559125934e-01 7.71943712957554317721786674155737e-02 1.64107847122030581221707734584925e-02 7.29979693943176750903223393152075e-03
            9.06394843992041510105650559125934e-01 1.64107847122030581221707734584925e-02 7.71943712957554317721786674155737e-02 7.29979693943176750903223393152075e-03
            7.71943712957554317721786674155737e-02 9.06394843992041510105650559125934e-01 1.64107847122030581221707734584925e-02 7.29979693943176750903223393152075e-03
            7.71943712957554317721786674155737e-02 1.64107847122030581221707734584925e-02 9.06394843992041510105650559125934e-01 7.29979693943176750903223393152075e-03
            1.64107847122030581221707734584925e-02 9.06394843992041510105650559125934e-01 7.71943712957554317721786674155737e-02 7.29979693943176750903223393152075e-03
            1.64107847122030581221707734584925e-02 7.71943712957554317721786674155737e-02 9.06394843992041510105650559125934e-01 7.29979693943176750903223393152075e-03
            ];
        
    case  17
        
        % ALG. DEG.:   17
        % PTS CARD.:   60
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.2e-16
        
        xyw_bar=[
            2.40563069636269027951058774306148e-01 2.40563069636269027951058774306148e-01 5.18873860727461888586731220129877e-01 3.82925400800356863229012560623232e-02
            2.40563069636269027951058774306148e-01 5.18873860727461888586731220129877e-01 2.40563069636269027951058774306148e-01 3.82925400800356863229012560623232e-02
            5.18873860727461888586731220129877e-01 2.40563069636269027951058774306148e-01 2.40563069636269027951058774306148e-01 3.82925400800356863229012560623232e-02
            8.09232358976607335510067287032143e-02 8.09232358976607335510067287032143e-02 8.38153528204678588409137773851398e-01 1.66952869977533974177319464615721e-02
            8.09232358976607335510067287032143e-02 8.38153528204678588409137773851398e-01 8.09232358976607335510067287032143e-02 1.66952869977533974177319464615721e-02
            8.38153528204678588409137773851398e-01 8.09232358976607335510067287032143e-02 8.09232358976607335510067287032143e-02 1.66952869977533974177319464615721e-02
            1.00141491249913505701130134184496e-02 1.00141491249913505701130134184496e-02 9.79971701750017354370925204420928e-01 1.43512454359061227167349006350605e-03
            1.00141491249913505701130134184496e-02 9.79971701750017354370925204420928e-01 1.00141491249913505701130134184496e-02 1.43512454359061227167349006350605e-03
            9.79971701750017354370925204420928e-01 1.00141491249913505701130134184496e-02 1.00141491249913505701130134184496e-02 1.43512454359061227167349006350605e-03
            1.54376520786632887016764925647294e-01 1.54376520786632887016764925647294e-01 6.91246958426734225966470148705412e-01 2.86427684918505350009088061824514e-02
            1.54376520786632887016764925647294e-01 6.91246958426734225966470148705412e-01 1.54376520786632887016764925647294e-01 2.86427684918505350009088061824514e-02
            6.91246958426734225966470148705412e-01 1.54376520786632887016764925647294e-01 1.54376520786632887016764925647294e-01 2.86427684918505350009088061824514e-02
            4.17169862019962700561137580734794e-01 4.17169862019962700561137580734794e-01 1.65660275960074598877724838530412e-01 3.40856907820621529703331020755286e-02
            4.17169862019962700561137580734794e-01 1.65660275960074598877724838530412e-01 4.17169862019962700561137580734794e-01 3.40856907820621529703331020755286e-02
            1.65660275960074598877724838530412e-01 4.17169862019962700561137580734794e-01 4.17169862019962700561137580734794e-01 3.40856907820621529703331020755286e-02
            4.70869745738400957701941251798416e-01 4.70869745738400957701941251798416e-01 5.82605085231980845961174964031670e-02 2.46727420005308922812403693569649e-02
            4.70869745738400957701941251798416e-01 5.82605085231980845961174964031670e-02 4.70869745738400957701941251798416e-01 2.46727420005308922812403693569649e-02
            5.82605085231980845961174964031670e-02 4.70869745738400957701941251798416e-01 4.70869745738400957701941251798416e-01 2.46727420005308922812403693569649e-02
            4.98118033845422025773785890123690e-01 4.98118033845422025773785890123690e-01 3.76393230915594845242821975261904e-03 5.86679757537134170114478948221404e-03
            4.98118033845422025773785890123690e-01 3.76393230915594845242821975261904e-03 4.98118033845422025773785890123690e-01 5.86679757537134170114478948221404e-03
            3.76393230915594845242821975261904e-03 4.98118033845422025773785890123690e-01 4.98118033845422025773785890123690e-01 5.86679757537134170114478948221404e-03
            3.64738405652919261967781494604424e-01 3.64738405652919261967781494604424e-01 2.70523188694161476064437010791153e-01 2.32185950042289625250280238333289e-02
            3.64738405652919261967781494604424e-01 2.70523188694161476064437010791153e-01 3.64738405652919261967781494604424e-01 2.32185950042289625250280238333289e-02
            2.70523188694161476064437010791153e-01 3.64738405652919261967781494604424e-01 3.64738405652919261967781494604424e-01 2.32185950042289625250280238333289e-02
            1.09865907082626154678273167064617e-01 3.04665769698665711562313163085491e-01 5.85468323218708119881625862035435e-01 3.08496545825140619512261253021279e-02
            1.09865907082626154678273167064617e-01 5.85468323218708119881625862035435e-01 3.04665769698665711562313163085491e-01 3.08496545825140619512261253021279e-02
            3.04665769698665711562313163085491e-01 1.09865907082626154678273167064617e-01 5.85468323218708119881625862035435e-01 3.08496545825140619512261253021279e-02
            3.04665769698665711562313163085491e-01 5.85468323218708119881625862035435e-01 1.09865907082626154678273167064617e-01 3.08496545825140619512261253021279e-02
            5.85468323218708119881625862035435e-01 1.09865907082626154678273167064617e-01 3.04665769698665711562313163085491e-01 3.08496545825140619512261253021279e-02
            5.85468323218708119881625862035435e-01 3.04665769698665711562313163085491e-01 1.09865907082626154678273167064617e-01 3.08496545825140619512261253021279e-02
            2.04932274629187893832948930139537e-01 5.24875839064542562550919058139698e-02 7.42580141464357801339701836695895e-01 1.88139854400541997603291122231894e-02
            2.04932274629187893832948930139537e-01 7.42580141464357801339701836695895e-01 5.24875839064542562550919058139698e-02 1.88139854400541997603291122231894e-02
            5.24875839064542562550919058139698e-02 2.04932274629187893832948930139537e-01 7.42580141464357801339701836695895e-01 1.88139854400541997603291122231894e-02
            5.24875839064542562550919058139698e-02 7.42580141464357801339701836695895e-01 2.04932274629187893832948930139537e-01 1.88139854400541997603291122231894e-02
            7.42580141464357801339701836695895e-01 2.04932274629187893832948930139537e-01 5.24875839064542562550919058139698e-02 1.88139854400541997603291122231894e-02
            7.42580141464357801339701836695895e-01 5.24875839064542562550919058139698e-02 2.04932274629187893832948930139537e-01 1.88139854400541997603291122231894e-02
            5.81392156426624381881929082283023e-02 1.50005399522595440214844941806405e-02 9.26860244405077970952788746217266e-01 5.12343450397285526259905097390401e-03
            5.81392156426624381881929082283023e-02 9.26860244405077970952788746217266e-01 1.50005399522595440214844941806405e-02 5.12343450397285526259905097390401e-03
            1.50005399522595440214844941806405e-02 5.81392156426624381881929082283023e-02 9.26860244405077970952788746217266e-01 5.12343450397285526259905097390401e-03
            1.50005399522595440214844941806405e-02 9.26860244405077970952788746217266e-01 5.81392156426624381881929082283023e-02 5.12343450397285526259905097390401e-03
            9.26860244405077970952788746217266e-01 5.81392156426624381881929082283023e-02 1.50005399522595440214844941806405e-02 5.12343450397285526259905097390401e-03
            9.26860244405077970952788746217266e-01 1.50005399522595440214844941806405e-02 5.81392156426624381881929082283023e-02 5.12343450397285526259905097390401e-03
            1.38595540867764821513219430926256e-01 1.50102334797318250070308209842551e-02 8.46394225652503329193621084414190e-01 7.01239348475201761806685496480895e-03
            1.38595540867764821513219430926256e-01 8.46394225652503329193621084414190e-01 1.50102334797318250070308209842551e-02 7.01239348475201761806685496480895e-03
            1.50102334797318250070308209842551e-02 1.38595540867764821513219430926256e-01 8.46394225652503329193621084414190e-01 7.01239348475201761806685496480895e-03
            1.50102334797318250070308209842551e-02 8.46394225652503329193621084414190e-01 1.38595540867764821513219430926256e-01 7.01239348475201761806685496480895e-03
            8.46394225652503329193621084414190e-01 1.38595540867764821513219430926256e-01 1.50102334797318250070308209842551e-02 7.01239348475201761806685496480895e-03
            8.46394225652503329193621084414190e-01 1.50102334797318250070308209842551e-02 1.38595540867764821513219430926256e-01 7.01239348475201761806685496480895e-03
            3.46605469520092579482906103294226e-01 2.33621289331465385730179917800342e-02 6.30032401546760878474628952972125e-01 1.53822944350446126626774656642738e-02
            3.46605469520092579482906103294226e-01 6.30032401546760878474628952972125e-01 2.33621289331465385730179917800342e-02 1.53822944350446126626774656642738e-02
            2.33621289331465385730179917800342e-02 3.46605469520092579482906103294226e-01 6.30032401546760878474628952972125e-01 1.53822944350446126626774656642738e-02
            2.33621289331465385730179917800342e-02 6.30032401546760878474628952972125e-01 3.46605469520092579482906103294226e-01 1.53822944350446126626774656642738e-02
            6.30032401546760878474628952972125e-01 3.46605469520092579482906103294226e-01 2.33621289331465385730179917800342e-02 1.53822944350446126626774656642738e-02
            6.30032401546760878474628952972125e-01 2.33621289331465385730179917800342e-02 3.46605469520092579482906103294226e-01 1.53822944350446126626774656642738e-02
            2.48219868895855927393157003280066e-01 9.99999999999657541679184274974190e-07 7.51779131104144071606754096137593e-01 3.03013148261713129855143833424336e-03
            2.48219868895855927393157003280066e-01 7.51779131104144071606754096137593e-01 9.99999999999657541679184274974190e-07 3.03013148261713129855143833424336e-03
            9.99999999999657541679184274974190e-07 2.48219868895855927393157003280066e-01 7.51779131104144071606754096137593e-01 3.03013148261713129855143833424336e-03
            9.99999999999657541679184274974190e-07 7.51779131104144071606754096137593e-01 2.48219868895855927393157003280066e-01 3.03013148261713129855143833424336e-03
            7.51779131104144071606754096137593e-01 2.48219868895855927393157003280066e-01 9.99999999999657541679184274974190e-07 3.03013148261713129855143833424336e-03
            7.51779131104144071606754096137593e-01 9.99999999999657541679184274974190e-07 2.48219868895855927393157003280066e-01 3.03013148261713129855143833424336e-03
            ];
        
    case  18
        
        % ALG. DEG.:   18
        % PTS CARD.:   73
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.0e-15
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.29063313889186651040041908800049e-02
            4.89609987073006347468862031746539e-01 4.89609987073006347468862031746539e-01 2.07800258539873050622759365069214e-02 1.03307318912720588754972084188921e-02
            4.89609987073006347468862031746539e-01 2.07800258539873050622759365069214e-02 4.89609987073006347468862031746539e-01 1.03307318912720588754972084188921e-02
            2.07800258539873050622759365069214e-02 4.89609987073006347468862031746539e-01 4.89609987073006347468862031746539e-01 1.03307318912720588754972084188921e-02
            4.54536892697892658254943398787873e-01 4.54536892697892658254943398787873e-01 9.09262146042146834901132024242543e-02 2.23872472630164028040145041131836e-02
            4.54536892697892658254943398787873e-01 9.09262146042146834901132024242543e-02 4.54536892697892658254943398787873e-01 2.23872472630164028040145041131836e-02
            9.09262146042146834901132024242543e-02 4.54536892697892658254943398787873e-01 4.54536892697892658254943398787873e-01 2.23872472630164028040145041131836e-02
            4.01416680649431201199917040867149e-01 4.01416680649431201199917040867149e-01 1.97166638701137597600165918265702e-01 3.02661258694680856962122561526485e-02
            4.01416680649431201199917040867149e-01 1.97166638701137597600165918265702e-01 4.01416680649431201199917040867149e-01 3.02661258694680856962122561526485e-02
            1.97166638701137597600165918265702e-01 4.01416680649431201199917040867149e-01 4.01416680649431201199917040867149e-01 3.02661258694680856962122561526485e-02
            2.55551654403097627810126368785859e-01 2.55551654403097627810126368785859e-01 4.88896691193804744379747262428282e-01 3.04909678021977950668208734441578e-02
            2.55551654403097627810126368785859e-01 4.88896691193804744379747262428282e-01 2.55551654403097627810126368785859e-01 3.04909678021977950668208734441578e-02
            4.88896691193804744379747262428282e-01 2.55551654403097627810126368785859e-01 2.55551654403097627810126368785859e-01 3.04909678021977950668208734441578e-02
            1.77077942152129547137562326497573e-01 1.77077942152129547137562326497573e-01 6.45844115695740850213724115747027e-01 2.41592127416409147011577829289308e-02
            1.77077942152129547137562326497573e-01 6.45844115695740850213724115747027e-01 1.77077942152129547137562326497573e-01 2.41592127416409147011577829289308e-02
            6.45844115695740850213724115747027e-01 1.77077942152129547137562326497573e-01 1.77077942152129547137562326497573e-01 2.41592127416409147011577829289308e-02
            1.10061053227951857746980124375114e-01 1.10061053227951857746980124375114e-01 7.79877893544096312261615366878686e-01 1.60508035868008810953533327392506e-02
            1.10061053227951857746980124375114e-01 7.79877893544096312261615366878686e-01 1.10061053227951857746980124375114e-01 1.60508035868008810953533327392506e-02
            7.79877893544096312261615366878686e-01 1.10061053227951857746980124375114e-01 1.10061053227951857746980124375114e-01 1.60508035868008810953533327392506e-02
            5.55286242518396727652074673642346e-02 5.55286242518396727652074673642346e-02 8.88942751496320626714009449642617e-01 8.08458026178406410910959323246061e-03
            5.55286242518396727652074673642346e-02 8.88942751496320626714009449642617e-01 5.55286242518396727652074673642346e-02 8.08458026178406410910959323246061e-03
            8.88942751496320626714009449642617e-01 5.55286242518396727652074673642346e-02 5.55286242518396727652074673642346e-02 8.08458026178406410910959323246061e-03
            1.26218637772286691040024564358646e-02 1.26218637772286691040024564358646e-02 9.74756272445542637505866423452972e-01 2.07936202748478177540492062291833e-03
            1.26218637772286691040024564358646e-02 9.74756272445542637505866423452972e-01 1.26218637772286691040024564358646e-02 2.07936202748478177540492062291833e-03
            9.74756272445542637505866423452972e-01 1.26218637772286691040024564358646e-02 1.26218637772286691040024564358646e-02 2.07936202748478177540492062291833e-03
            6.00633794794644959580409704358317e-01 3.95754787356942860920838711535907e-01 3.61141784841217949875158410577569e-03 3.88487690498139168898683593056376e-03
            6.00633794794644959580409704358317e-01 3.61141784841217949875158410577569e-03 3.95754787356942860920838711535907e-01 3.88487690498139168898683593056376e-03
            3.95754787356942860920838711535907e-01 6.00633794794644959580409704358317e-01 3.61141784841217949875158410577569e-03 3.88487690498139168898683593056376e-03
            3.95754787356942860920838711535907e-01 3.61141784841217949875158410577569e-03 6.00633794794644959580409704358317e-01 3.88487690498139168898683593056376e-03
            3.61141784841217949875158410577569e-03 6.00633794794644959580409704358317e-01 3.95754787356942860920838711535907e-01 3.88487690498139168898683593056376e-03
            3.61141784841217949875158410577569e-03 3.95754787356942860920838711535907e-01 6.00633794794644959580409704358317e-01 3.88487690498139168898683593056376e-03
            1.34466754530779775134163855909719e-01 5.57603261588783927216184110875474e-01 3.07929983880436242138500801956980e-01 2.55741606120219143460925437238984e-02
            1.34466754530779775134163855909719e-01 3.07929983880436242138500801956980e-01 5.57603261588783927216184110875474e-01 2.55741606120219143460925437238984e-02
            5.57603261588783927216184110875474e-01 1.34466754530779775134163855909719e-01 3.07929983880436242138500801956980e-01 2.55741606120219143460925437238984e-02
            5.57603261588783927216184110875474e-01 3.07929983880436242138500801956980e-01 1.34466754530779775134163855909719e-01 2.55741606120219143460925437238984e-02
            3.07929983880436242138500801956980e-01 1.34466754530779775134163855909719e-01 5.57603261588783927216184110875474e-01 2.55741606120219143460925437238984e-02
            3.07929983880436242138500801956980e-01 5.57603261588783927216184110875474e-01 1.34466754530779775134163855909719e-01 2.55741606120219143460925437238984e-02
            7.20987025817365068114384030195652e-01 2.64566948406520230374638913417584e-01 1.44460257761147015109770563867642e-02 8.88090357333806180994972834241707e-03
            7.20987025817365068114384030195652e-01 1.44460257761147015109770563867642e-02 2.64566948406520230374638913417584e-01 8.88090357333806180994972834241707e-03
            2.64566948406520230374638913417584e-01 7.20987025817365068114384030195652e-01 1.44460257761147015109770563867642e-02 8.88090357333806180994972834241707e-03
            2.64566948406520230374638913417584e-01 1.44460257761147015109770563867642e-02 7.20987025817365068114384030195652e-01 8.88090357333806180994972834241707e-03
            1.44460257761147015109770563867642e-02 7.20987025817365068114384030195652e-01 2.64566948406520230374638913417584e-01 8.88090357333806180994972834241707e-03
            1.44460257761147015109770563867642e-02 2.64566948406520230374638913417584e-01 7.20987025817365068114384030195652e-01 8.88090357333806180994972834241707e-03
            5.94527068955870929833906757266959e-01 3.58539352205950567764602965326048e-01 4.69335788381785024014902774069924e-02 1.61245467617313967267911323233420e-02
            5.94527068955870929833906757266959e-01 4.69335788381785024014902774069924e-02 3.58539352205950567764602965326048e-01 1.61245467617313967267911323233420e-02
            3.58539352205950567764602965326048e-01 5.94527068955870929833906757266959e-01 4.69335788381785024014902774069924e-02 1.61245467617313967267911323233420e-02
            3.58539352205950567764602965326048e-01 4.69335788381785024014902774069924e-02 5.94527068955870929833906757266959e-01 1.61245467617313967267911323233420e-02
            4.69335788381785024014902774069924e-02 5.94527068955870929833906757266959e-01 3.58539352205950567764602965326048e-01 1.61245467617313967267911323233420e-02
            4.69335788381785024014902774069924e-02 3.58539352205950567764602965326048e-01 5.94527068955870929833906757266959e-01 1.61245467617313967267911323233420e-02
            8.39331473680838602291487404727377e-01 1.57807405968594743894684029328346e-01 2.86112035056665381382856594427722e-03 2.49194181749067683595555422471080e-03
            8.39331473680838602291487404727377e-01 2.86112035056665381382856594427722e-03 1.57807405968594743894684029328346e-01 2.49194181749067683595555422471080e-03
            1.57807405968594743894684029328346e-01 8.39331473680838602291487404727377e-01 2.86112035056665381382856594427722e-03 2.49194181749067683595555422471080e-03
            1.57807405968594743894684029328346e-01 2.86112035056665381382856594427722e-03 8.39331473680838602291487404727377e-01 2.49194181749067683595555422471080e-03
            2.86112035056665381382856594427722e-03 8.39331473680838602291487404727377e-01 1.57807405968594743894684029328346e-01 2.49194181749067683595555422471080e-03
            2.86112035056665381382856594427722e-03 1.57807405968594743894684029328346e-01 8.39331473680838602291487404727377e-01 2.49194181749067683595555422471080e-03
            2.23861424097915678865788891016564e-01 7.01087978926173360783025145792635e-01 7.50505969759109881067615788197145e-02 1.82428401189505852431071986075040e-02
            2.23861424097915678865788891016564e-01 7.50505969759109881067615788197145e-02 7.01087978926173360783025145792635e-01 1.82428401189505852431071986075040e-02
            7.01087978926173360783025145792635e-01 2.23861424097915678865788891016564e-01 7.50505969759109881067615788197145e-02 1.82428401189505852431071986075040e-02
            7.01087978926173360783025145792635e-01 7.50505969759109881067615788197145e-02 2.23861424097915678865788891016564e-01 1.82428401189505852431071986075040e-02
            7.50505969759109881067615788197145e-02 2.23861424097915678865788891016564e-01 7.01087978926173360783025145792635e-01 1.82428401189505852431071986075040e-02
            7.50505969759109881067615788197145e-02 7.01087978926173360783025145792635e-01 2.23861424097915678865788891016564e-01 1.82428401189505852431071986075040e-02
            8.22931324069856584557669521018397e-01 1.42421601113383450298854882021260e-01 3.46470748167599651434755969603430e-02 1.02585637361985258092511585914508e-02
            8.22931324069856584557669521018397e-01 3.46470748167599651434755969603430e-02 1.42421601113383450298854882021260e-01 1.02585637361985258092511585914508e-02
            1.42421601113383450298854882021260e-01 8.22931324069856584557669521018397e-01 3.46470748167599651434755969603430e-02 1.02585637361985258092511585914508e-02
            1.42421601113383450298854882021260e-01 3.46470748167599651434755969603430e-02 8.22931324069856584557669521018397e-01 1.02585637361985258092511585914508e-02
            3.46470748167599651434755969603430e-02 8.22931324069856584557669521018397e-01 1.42421601113383450298854882021260e-01 1.02585637361985258092511585914508e-02
            3.46470748167599651434755969603430e-02 1.42421601113383450298854882021260e-01 8.22931324069856584557669521018397e-01 1.02585637361985258092511585914508e-02
            9.24344252620784012641763638384873e-01 6.54946280829376975551880946113670e-02 1.01611192962782898030482670037600e-02 3.79992885530191582335590716468232e-03
            9.24344252620784012641763638384873e-01 1.01611192962782898030482670037600e-02 6.54946280829376975551880946113670e-02 3.79992885530191582335590716468232e-03
            6.54946280829376975551880946113670e-02 9.24344252620784012641763638384873e-01 1.01611192962782898030482670037600e-02 3.79992885530191582335590716468232e-03
            6.54946280829376975551880946113670e-02 1.01611192962782898030482670037600e-02 9.24344252620784012641763638384873e-01 3.79992885530191582335590716468232e-03
            1.01611192962782898030482670037600e-02 9.24344252620784012641763638384873e-01 6.54946280829376975551880946113670e-02 3.79992885530191582335590716468232e-03
            1.01611192962782898030482670037600e-02 6.54946280829376975551880946113670e-02 9.24344252620784012641763638384873e-01 3.79992885530191582335590716468232e-03
            ];
        
    case  20
        
        % ALG. DEG.:   20
        % PTS CARD.:   82
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 9.1e-16
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 2.34389883762168745040543882396378e-02
            4.72539690493749442445903241605265e-01 4.72539690493749442445903241605265e-01 5.49206190125011151081935167894699e-02 1.70118788706518052344929259334094e-02
            4.72539690493749442445903241605265e-01 5.49206190125011151081935167894699e-02 4.72539690493749442445903241605265e-01 1.70118788706518052344929259334094e-02
            5.49206190125011151081935167894699e-02 4.72539690493749442445903241605265e-01 4.72539690493749442445903241605265e-01 1.70118788706518052344929259334094e-02
            4.35591797658190560404989355447469e-01 4.35591797658190560404989355447469e-01 1.28816404683618879190021289105061e-01 2.21346290253984917528740083980665e-02
            4.35591797658190560404989355447469e-01 1.28816404683618879190021289105061e-01 4.35591797658190560404989355447469e-01 2.21346290253984917528740083980665e-02
            1.28816404683618879190021289105061e-01 4.35591797658190560404989355447469e-01 4.35591797658190560404989355447469e-01 2.21346290253984917528740083980665e-02
            3.85483177690954081917595885897754e-01 3.85483177690954081917595885897754e-01 2.29033644618091836164808228204492e-01 2.22501203414893870824453614432059e-02
            3.85483177690954081917595885897754e-01 2.29033644618091836164808228204492e-01 3.85483177690954081917595885897754e-01 2.22501203414893870824453614432059e-02
            2.29033644618091836164808228204492e-01 3.85483177690954081917595885897754e-01 3.85483177690954081917595885897754e-01 2.22501203414893870824453614432059e-02
            1.85897872669382596688691933195514e-01 1.85897872669382596688691933195514e-01 6.28204254661234751111464902351145e-01 2.10880142751876734075189290251728e-02
            1.85897872669382596688691933195514e-01 6.28204254661234751111464902351145e-01 1.85897872669382596688691933195514e-01 2.10880142751876734075189290251728e-02
            6.28204254661234751111464902351145e-01 1.85897872669382596688691933195514e-01 1.85897872669382596688691933195514e-01 2.10880142751876734075189290251728e-02
            1.02943093872275681643913003426860e-01 1.02943093872275681643913003426860e-01 7.94113812255448636712173993146280e-01 1.43667319823725190414132057981078e-02
            1.02943093872275681643913003426860e-01 7.94113812255448636712173993146280e-01 1.02943093872275681643913003426860e-01 1.43667319823725190414132057981078e-02
            7.94113812255448636712173993146280e-01 1.02943093872275681643913003426860e-01 1.02943093872275681643913003426860e-01 1.43667319823725190414132057981078e-02
            4.42043568221049953947954236355145e-02 4.42043568221049953947954236355145e-02 9.11591286355790009210409152728971e-01 5.96064247309055408180489266101176e-03
            4.42043568221049953947954236355145e-02 9.11591286355790009210409152728971e-01 4.42043568221049953947954236355145e-02 5.96064247309055408180489266101176e-03
            9.11591286355790009210409152728971e-01 4.42043568221049953947954236355145e-02 4.42043568221049953947954236355145e-02 5.96064247309055408180489266101176e-03
            1.18770000819499090688458053932663e-02 1.18770000819499090688458053932663e-02 9.76245999836100164515073629445396e-01 1.85446185638856396969698092647150e-03
            1.18770000819499090688458053932663e-02 9.76245999836100164515073629445396e-01 1.18770000819499090688458053932663e-02 1.85446185638856396969698092647150e-03
            9.76245999836100164515073629445396e-01 1.18770000819499090688458053932663e-02 1.18770000819499090688458053932663e-02 1.85446185638856396969698092647150e-03
            4.23177101943935984884603840328055e-01 5.66246814212167381974438740144251e-01 1.05760838438966331409574195276946e-02 7.66503799888174162241449849375385e-03
            4.23177101943935984884603840328055e-01 1.05760838438966331409574195276946e-02 5.66246814212167381974438740144251e-01 7.66503799888174162241449849375385e-03
            5.66246814212167381974438740144251e-01 4.23177101943935984884603840328055e-01 1.05760838438966331409574195276946e-02 7.66503799888174162241449849375385e-03
            5.66246814212167381974438740144251e-01 1.05760838438966331409574195276946e-02 4.23177101943935984884603840328055e-01 7.66503799888174162241449849375385e-03
            1.05760838438966331409574195276946e-02 4.23177101943935984884603840328055e-01 5.66246814212167381974438740144251e-01 7.66503799888174162241449849375385e-03
            1.05760838438966331409574195276946e-02 5.66246814212167381974438740144251e-01 4.23177101943935984884603840328055e-01 7.66503799888174162241449849375385e-03
            2.87178034937351334132671354382182e-01 7.03631343750081739152335558173945e-01 9.19062131256687120384185618604533e-03 6.11096641377270079209527153807358e-03
            2.87178034937351334132671354382182e-01 9.19062131256687120384185618604533e-03 7.03631343750081739152335558173945e-01 6.11096641377270079209527153807358e-03
            7.03631343750081739152335558173945e-01 2.87178034937351334132671354382182e-01 9.19062131256687120384185618604533e-03 6.11096641377270079209527153807358e-03
            7.03631343750081739152335558173945e-01 9.19062131256687120384185618604533e-03 2.87178034937351334132671354382182e-01 6.11096641377270079209527153807358e-03
            9.19062131256687120384185618604533e-03 2.87178034937351334132671354382182e-01 7.03631343750081739152335558173945e-01 6.11096641377270079209527153807358e-03
            9.19062131256687120384185618604533e-03 7.03631343750081739152335558173945e-01 2.87178034937351334132671354382182e-01 6.11096641377270079209527153807358e-03
            1.62824865710703176047147167082585e-01 8.32373311160547357800965073693078e-01 4.80182312874943839631214359542355e-03 3.29750122890750665613501801942675e-03
            1.62824865710703176047147167082585e-01 4.80182312874943839631214359542355e-03 8.32373311160547357800965073693078e-01 3.29750122890750665613501801942675e-03
            8.32373311160547357800965073693078e-01 1.62824865710703176047147167082585e-01 4.80182312874943839631214359542355e-03 3.29750122890750665613501801942675e-03
            8.32373311160547357800965073693078e-01 4.80182312874943839631214359542355e-03 1.62824865710703176047147167082585e-01 3.29750122890750665613501801942675e-03
            4.80182312874943839631214359542355e-03 1.62824865710703176047147167082585e-01 8.32373311160547357800965073693078e-01 3.29750122890750665613501801942675e-03
            4.80182312874943839631214359542355e-03 8.32373311160547357800965073693078e-01 1.62824865710703176047147167082585e-01 3.29750122890750665613501801942675e-03
            6.46806344981751130118752257658343e-02 9.29257699458375552303834865597310e-01 6.06166604344937631765333208022639e-03 2.58504328745784439647081853763666e-03
            6.46806344981751130118752257658343e-02 6.06166604344937631765333208022639e-03 9.29257699458375552303834865597310e-01 2.58504328745784439647081853763666e-03
            9.29257699458375552303834865597310e-01 6.46806344981751130118752257658343e-02 6.06166604344937631765333208022639e-03 2.58504328745784439647081853763666e-03
            9.29257699458375552303834865597310e-01 6.06166604344937631765333208022639e-03 6.46806344981751130118752257658343e-02 2.58504328745784439647081853763666e-03
            6.06166604344937631765333208022639e-03 6.46806344981751130118752257658343e-02 9.29257699458375552303834865597310e-01 2.58504328745784439647081853763666e-03
            6.06166604344937631765333208022639e-03 9.29257699458375552303834865597310e-01 6.46806344981751130118752257658343e-02 2.58504328745784439647081853763666e-03
            3.34006277005119089906770568632055e-01 6.13237016399387990794878078304464e-01 5.27567065954929192983513530634809e-02 1.60482996634683516834485317303916e-02
            3.34006277005119089906770568632055e-01 5.27567065954929192983513530634809e-02 6.13237016399387990794878078304464e-01 1.60482996634683516834485317303916e-02
            6.13237016399387990794878078304464e-01 3.34006277005119089906770568632055e-01 5.27567065954929192983513530634809e-02 1.60482996634683516834485317303916e-02
            6.13237016399387990794878078304464e-01 5.27567065954929192983513530634809e-02 3.34006277005119089906770568632055e-01 1.60482996634683516834485317303916e-02
            5.27567065954929192983513530634809e-02 3.34006277005119089906770568632055e-01 6.13237016399387990794878078304464e-01 1.60482996634683516834485317303916e-02
            5.27567065954929192983513530634809e-02 6.13237016399387990794878078304464e-01 3.34006277005119089906770568632055e-01 1.60482996634683516834485317303916e-02
            2.10923795524181278970843322895234e-01 7.51510514906010507196754133474315e-01 3.75656895698082138324025436304510e-02 9.87379147789447554617314750657897e-03
            2.10923795524181278970843322895234e-01 3.75656895698082138324025436304510e-02 7.51510514906010507196754133474315e-01 9.87379147789447554617314750657897e-03
            7.51510514906010507196754133474315e-01 2.10923795524181278970843322895234e-01 3.75656895698082138324025436304510e-02 9.87379147789447554617314750657897e-03
            7.51510514906010507196754133474315e-01 3.75656895698082138324025436304510e-02 2.10923795524181278970843322895234e-01 9.87379147789447554617314750657897e-03
            3.75656895698082138324025436304510e-02 2.10923795524181278970843322895234e-01 7.51510514906010507196754133474315e-01 9.87379147789447554617314750657897e-03
            3.75656895698082138324025436304510e-02 7.51510514906010507196754133474315e-01 2.10923795524181278970843322895234e-01 9.87379147789447554617314750657897e-03
            1.15080380841368318245443447267462e-01 8.50151131562102868599595240084454e-01 3.47684875965288409105369282769971e-02 8.17314741769008777649840880030752e-03
            1.15080380841368318245443447267462e-01 3.47684875965288409105369282769971e-02 8.50151131562102868599595240084454e-01 8.17314741769008777649840880030752e-03
            8.50151131562102868599595240084454e-01 1.15080380841368318245443447267462e-01 3.47684875965288409105369282769971e-02 8.17314741769008777649840880030752e-03
            8.50151131562102868599595240084454e-01 3.47684875965288409105369282769971e-02 1.15080380841368318245443447267462e-01 8.17314741769008777649840880030752e-03
            3.47684875965288409105369282769971e-02 1.15080380841368318245443447267462e-01 8.50151131562102868599595240084454e-01 8.17314741769008777649840880030752e-03
            3.47684875965288409105369282769971e-02 8.50151131562102868599595240084454e-01 1.15080380841368318245443447267462e-01 8.17314741769008777649840880030752e-03
            3.10752086465084287336679835789255e-01 5.61989749539536176392573452176293e-01 1.27258163995379480759595480776625e-01 2.12721242081270953128147027655359e-02
            3.10752086465084287336679835789255e-01 1.27258163995379480759595480776625e-01 5.61989749539536176392573452176293e-01 2.12721242081270953128147027655359e-02
            5.61989749539536176392573452176293e-01 3.10752086465084287336679835789255e-01 1.27258163995379480759595480776625e-01 2.12721242081270953128147027655359e-02
            5.61989749539536176392573452176293e-01 1.27258163995379480759595480776625e-01 3.10752086465084287336679835789255e-01 2.12721242081270953128147027655359e-02
            1.27258163995379480759595480776625e-01 3.10752086465084287336679835789255e-01 5.61989749539536176392573452176293e-01 2.12721242081270953128147027655359e-02
            1.27258163995379480759595480776625e-01 5.61989749539536176392573452176293e-01 3.10752086465084287336679835789255e-01 2.12721242081270953128147027655359e-02
            2.02006198010458992708748837685562e-01 7.01716440236161864696384782291716e-01 9.62773617533790870837151487648953e-02 1.62955996807267987358525118679609e-02
            2.02006198010458992708748837685562e-01 9.62773617533790870837151487648953e-02 7.01716440236161864696384782291716e-01 1.62955996807267987358525118679609e-02
            7.01716440236161864696384782291716e-01 2.02006198010458992708748837685562e-01 9.62773617533790870837151487648953e-02 1.62955996807267987358525118679609e-02
            7.01716440236161864696384782291716e-01 9.62773617533790870837151487648953e-02 2.02006198010458992708748837685562e-01 1.62955996807267987358525118679609e-02
            9.62773617533790870837151487648953e-02 2.02006198010458992708748837685562e-01 7.01716440236161864696384782291716e-01 1.62955996807267987358525118679609e-02
            9.62773617533790870837151487648953e-02 7.01716440236161864696384782291716e-01 2.02006198010458992708748837685562e-01 1.62955996807267987358525118679609e-02
            2.89023207908959300915796575282002e-01 4.91388562323192112923919694367214e-01 2.19588229767848586160283730350784e-01 1.91054178147478971594441787829055e-02
            2.89023207908959300915796575282002e-01 2.19588229767848586160283730350784e-01 4.91388562323192112923919694367214e-01 1.91054178147478971594441787829055e-02
            4.91388562323192112923919694367214e-01 2.89023207908959300915796575282002e-01 2.19588229767848586160283730350784e-01 1.91054178147478971594441787829055e-02
            4.91388562323192112923919694367214e-01 2.19588229767848586160283730350784e-01 2.89023207908959300915796575282002e-01 1.91054178147478971594441787829055e-02
            2.19588229767848586160283730350784e-01 2.89023207908959300915796575282002e-01 4.91388562323192112923919694367214e-01 1.91054178147478971594441787829055e-02
            2.19588229767848586160283730350784e-01 4.91388562323192112923919694367214e-01 2.89023207908959300915796575282002e-01 1.91054178147478971594441787829055e-02
            ];
        
    case  21
        
        % ALG. DEG.:   21
        % PTS CARD.:   87
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.6e-16
        
        xyw_bar=[
            1.08306479138961346542213703969537e-02 1.08306479138961346542213703969537e-02 9.78338704172207695997087739669951e-01 1.53252214157029989158309568608729e-03
            1.08306479138961346542213703969537e-02 9.78338704172207695997087739669951e-01 1.08306479138961346542213703969537e-02 1.53252214157029989158309568608729e-03
            9.78338704172207695997087739669951e-01 1.08306479138961346542213703969537e-02 1.08306479138961346542213703969537e-02 1.53252214157029989158309568608729e-03
            4.98681169280067615456886187530472e-01 4.98681169280067615456886187530472e-01 2.63766143986476908622762493905611e-03 2.95964208580089174066563018072884e-03
            4.98681169280067615456886187530472e-01 2.63766143986476908622762493905611e-03 4.98681169280067615456886187530472e-01 2.95964208580089174066563018072884e-03
            2.63766143986476908622762493905611e-03 4.98681169280067615456886187530472e-01 4.98681169280067615456886187530472e-01 2.95964208580089174066563018072884e-03
            5.32063025937022163192047230495518e-02 5.32063025937022163192047230495518e-02 8.93587394812595525728227130457526e-01 7.10507195198323022317099528777362e-03
            5.32063025937022163192047230495518e-02 8.93587394812595525728227130457526e-01 5.32063025937022163192047230495518e-02 7.10507195198323022317099528777362e-03
            8.93587394812595525728227130457526e-01 5.32063025937022163192047230495518e-02 5.32063025937022163192047230495518e-02 7.10507195198323022317099528777362e-03
            4.84368806799071960345060006147833e-01 4.84368806799071960345060006147833e-01 3.12623864018560793098799877043348e-02 1.24220254501511973010963174601784e-02
            4.84368806799071960345060006147833e-01 3.12623864018560793098799877043348e-02 4.84368806799071960345060006147833e-01 1.24220254501511973010963174601784e-02
            3.12623864018560793098799877043348e-02 4.84368806799071960345060006147833e-01 4.84368806799071960345060006147833e-01 1.24220254501511973010963174601784e-02
            1.17179324464095319102696635127359e-01 1.17179324464095319102696635127359e-01 7.65641351071809417305757961003110e-01 1.31538530905486086147737623264220e-02
            1.17179324464095319102696635127359e-01 7.65641351071809417305757961003110e-01 1.17179324464095319102696635127359e-01 1.31538530905486086147737623264220e-02
            7.65641351071809417305757961003110e-01 1.17179324464095319102696635127359e-01 1.17179324464095319102696635127359e-01 1.31538530905486086147737623264220e-02
            1.86199874531600445548562561270955e-01 1.86199874531600445548562561270955e-01 6.27600250936799053391723646200262e-01 1.61071199335594168744467680198795e-02
            1.86199874531600445548562561270955e-01 6.27600250936799053391723646200262e-01 1.86199874531600445548562561270955e-01 1.61071199335594168744467680198795e-02
            6.27600250936799053391723646200262e-01 1.86199874531600445548562561270955e-01 1.86199874531600445548562561270955e-01 1.61071199335594168744467680198795e-02
            2.37705046970502376080247586287442e-01 2.37705046970502376080247586287442e-01 5.24589906058995247839504827425117e-01 1.95569983274215403179496775010193e-02
            2.37705046970502376080247586287442e-01 5.24589906058995247839504827425117e-01 2.37705046970502376080247586287442e-01 1.95569983274215403179496775010193e-02
            5.24589906058995247839504827425117e-01 2.37705046970502376080247586287442e-01 2.37705046970502376080247586287442e-01 1.95569983274215403179496775010193e-02
            4.51118096863331430323995618891786e-01 4.51118096863331430323995618891786e-01 9.77638062733371393520087622164283e-02 2.11471458034111631230089756172674e-02
            4.51118096863331430323995618891786e-01 9.77638062733371393520087622164283e-02 4.51118096863331430323995618891786e-01 2.11471458034111631230089756172674e-02
            9.77638062733371393520087622164283e-02 4.51118096863331430323995618891786e-01 4.51118096863331430323995618891786e-01 2.11471458034111631230089756172674e-02
            2.98901616618379606382660540475626e-01 2.98901616618379606382660540475626e-01 4.02196766763240787234678919048747e-01 2.14872496180955434297299433410444e-02
            2.98901616618379606382660540475626e-01 4.02196766763240787234678919048747e-01 2.98901616618379606382660540475626e-01 2.14872496180955434297299433410444e-02
            4.02196766763240787234678919048747e-01 2.98901616618379606382660540475626e-01 2.98901616618379606382660540475626e-01 2.14872496180955434297299433410444e-02
            1.01703755139702384929778489208729e-02 5.61060787404411287493388726943522e-02 9.33723545745588578981255523103755e-01 3.24321603629715987660020459770749e-03
            1.01703755139702384929778489208729e-02 9.33723545745588578981255523103755e-01 5.61060787404411287493388726943522e-02 3.24321603629715987660020459770749e-03
            5.61060787404411287493388726943522e-02 1.01703755139702384929778489208729e-02 9.33723545745588578981255523103755e-01 3.24321603629715987660020459770749e-03
            5.61060787404411287493388726943522e-02 9.33723545745588578981255523103755e-01 1.01703755139702384929778489208729e-02 3.24321603629715987660020459770749e-03
            9.33723545745588578981255523103755e-01 1.01703755139702384929778489208729e-02 5.61060787404411287493388726943522e-02 3.24321603629715987660020459770749e-03
            9.33723545745588578981255523103755e-01 5.61060787404411287493388726943522e-02 1.01703755139702384929778489208729e-02 3.24321603629715987660020459770749e-03
            7.40148001912942655328153662708246e-03 2.38060908939278198515054896233778e-01 7.54537611041592337635108833637787e-01 4.44606682456729750824164781874970e-03
            7.40148001912942655328153662708246e-03 7.54537611041592337635108833637787e-01 2.38060908939278198515054896233778e-01 4.44606682456729750824164781874970e-03
            2.38060908939278198515054896233778e-01 7.40148001912942655328153662708246e-03 7.54537611041592337635108833637787e-01 4.44606682456729750824164781874970e-03
            2.38060908939278198515054896233778e-01 7.54537611041592337635108833637787e-01 7.40148001912942655328153662708246e-03 4.44606682456729750824164781874970e-03
            7.54537611041592337635108833637787e-01 7.40148001912942655328153662708246e-03 2.38060908939278198515054896233778e-01 4.44606682456729750824164781874970e-03
            7.54537611041592337635108833637787e-01 2.38060908939278198515054896233778e-01 7.40148001912942655328153662708246e-03 4.44606682456729750824164781874970e-03
            9.48888021003903639472021325218520e-03 1.33714236127292523947929225869302e-01 8.56796883662668484760160936275497e-01 4.44713701276898756842292925739457e-03
            9.48888021003903639472021325218520e-03 8.56796883662668484760160936275497e-01 1.33714236127292523947929225869302e-01 4.44713701276898756842292925739457e-03
            1.33714236127292523947929225869302e-01 9.48888021003903639472021325218520e-03 8.56796883662668484760160936275497e-01 4.44713701276898756842292925739457e-03
            1.33714236127292523947929225869302e-01 8.56796883662668484760160936275497e-01 9.48888021003903639472021325218520e-03 4.44713701276898756842292925739457e-03
            8.56796883662668484760160936275497e-01 9.48888021003903639472021325218520e-03 1.33714236127292523947929225869302e-01 4.44713701276898756842292925739457e-03
            8.56796883662668484760160936275497e-01 1.33714236127292523947929225869302e-01 9.48888021003903639472021325218520e-03 4.44713701276898756842292925739457e-03
            1.07870852895329272236324413825059e-02 3.60878270361336406235608365022927e-01 6.28334644349130666540759193594567e-01 7.19283008979440616897926830120014e-03
            1.07870852895329272236324413825059e-02 6.28334644349130666540759193594567e-01 3.60878270361336406235608365022927e-01 7.19283008979440616897926830120014e-03
            3.60878270361336406235608365022927e-01 1.07870852895329272236324413825059e-02 6.28334644349130666540759193594567e-01 7.19283008979440616897926830120014e-03
            3.60878270361336406235608365022927e-01 6.28334644349130666540759193594567e-01 1.07870852895329272236324413825059e-02 7.19283008979440616897926830120014e-03
            6.28334644349130666540759193594567e-01 1.07870852895329272236324413825059e-02 3.60878270361336406235608365022927e-01 7.19283008979440616897926830120014e-03
            6.28334644349130666540759193594567e-01 3.60878270361336406235608365022927e-01 1.07870852895329272236324413825059e-02 7.19283008979440616897926830120014e-03
            4.88864101564119649112605259233533e-02 1.28560007074346699385003489624069e-01 8.22553582769241398153781119617634e-01 9.68818276967736181914947479754119e-03
            4.88864101564119649112605259233533e-02 8.22553582769241398153781119617634e-01 1.28560007074346699385003489624069e-01 9.68818276967736181914947479754119e-03
            1.28560007074346699385003489624069e-01 4.88864101564119649112605259233533e-02 8.22553582769241398153781119617634e-01 9.68818276967736181914947479754119e-03
            1.28560007074346699385003489624069e-01 8.22553582769241398153781119617634e-01 4.88864101564119649112605259233533e-02 9.68818276967736181914947479754119e-03
            8.22553582769241398153781119617634e-01 4.88864101564119649112605259233533e-02 1.28560007074346699385003489624069e-01 9.68818276967736181914947479754119e-03
            8.22553582769241398153781119617634e-01 1.28560007074346699385003489624069e-01 4.88864101564119649112605259233533e-02 9.68818276967736181914947479754119e-03
            4.07904784349213886396334771688998e-02 2.31929220198316188294995754404226e-01 7.27280301366762471637628095777472e-01 1.07718672547460099508942477086748e-02
            4.07904784349213886396334771688998e-02 7.27280301366762471637628095777472e-01 2.31929220198316188294995754404226e-01 1.07718672547460099508942477086748e-02
            2.31929220198316188294995754404226e-01 4.07904784349213886396334771688998e-02 7.27280301366762471637628095777472e-01 1.07718672547460099508942477086748e-02
            2.31929220198316188294995754404226e-01 7.27280301366762471637628095777472e-01 4.07904784349213886396334771688998e-02 1.07718672547460099508942477086748e-02
            7.27280301366762471637628095777472e-01 4.07904784349213886396334771688998e-02 2.31929220198316188294995754404226e-01 1.07718672547460099508942477086748e-02
            7.27280301366762471637628095777472e-01 2.31929220198316188294995754404226e-01 4.07904784349213886396334771688998e-02 1.07718672547460099508942477086748e-02
            1.02254765254272128593981960875681e-01 2.13182380246852126726153642266581e-01 6.84562854498875772435440012486652e-01 1.53764487922742019987421002724659e-02
            1.02254765254272128593981960875681e-01 6.84562854498875772435440012486652e-01 2.13182380246852126726153642266581e-01 1.53764487922742019987421002724659e-02
            2.13182380246852126726153642266581e-01 1.02254765254272128593981960875681e-01 6.84562854498875772435440012486652e-01 1.53764487922742019987421002724659e-02
            2.13182380246852126726153642266581e-01 6.84562854498875772435440012486652e-01 1.02254765254272128593981960875681e-01 1.53764487922742019987421002724659e-02
            6.84562854498875772435440012486652e-01 1.02254765254272128593981960875681e-01 2.13182380246852126726153642266581e-01 1.53764487922742019987421002724659e-02
            6.84562854498875772435440012486652e-01 2.13182380246852126726153642266581e-01 1.02254765254272128593981960875681e-01 1.53764487922742019987421002724659e-02
            5.61951037264934380455194684600428e-02 3.46153638930814133711777458302095e-01 5.97651257342692421303809169330634e-01 1.54182016250510034449661134203780e-02
            5.61951037264934380455194684600428e-02 5.97651257342692421303809169330634e-01 3.46153638930814133711777458302095e-01 1.54182016250510034449661134203780e-02
            3.46153638930814133711777458302095e-01 5.61951037264934380455194684600428e-02 5.97651257342692421303809169330634e-01 1.54182016250510034449661134203780e-02
            3.46153638930814133711777458302095e-01 5.97651257342692421303809169330634e-01 5.61951037264934380455194684600428e-02 1.54182016250510034449661134203780e-02
            5.97651257342692421303809169330634e-01 5.61951037264934380455194684600428e-02 3.46153638930814133711777458302095e-01 1.54182016250510034449661134203780e-02
            5.97651257342692421303809169330634e-01 3.46153638930814133711777458302095e-01 5.61951037264934380455194684600428e-02 1.54182016250510034449661134203780e-02
            1.93921114022049395630986623473291e-01 3.57417794125976018282386803548434e-01 4.48661091851974613842202188607189e-01 1.87279940818234971744438155383250e-02
            1.93921114022049395630986623473291e-01 4.48661091851974613842202188607189e-01 3.57417794125976018282386803548434e-01 1.87279940818234971744438155383250e-02
            3.57417794125976018282386803548434e-01 1.93921114022049395630986623473291e-01 4.48661091851974613842202188607189e-01 1.87279940818234971744438155383250e-02
            3.57417794125976018282386803548434e-01 4.48661091851974613842202188607189e-01 1.93921114022049395630986623473291e-01 1.87279940818234971744438155383250e-02
            4.48661091851974613842202188607189e-01 1.93921114022049395630986623473291e-01 3.57417794125976018282386803548434e-01 1.87279940818234971744438155383250e-02
            4.48661091851974613842202188607189e-01 3.57417794125976018282386803548434e-01 1.93921114022049395630986623473291e-01 1.87279940818234971744438155383250e-02
            1.31396937599549262776221780768537e-01 3.09641800485391083253006172526511e-01 5.58961261915059681726347662333865e-01 1.96189079783958726654891080443122e-02
            1.31396937599549262776221780768537e-01 5.58961261915059681726347662333865e-01 3.09641800485391083253006172526511e-01 1.96189079783958726654891080443122e-02
            3.09641800485391083253006172526511e-01 1.31396937599549262776221780768537e-01 5.58961261915059681726347662333865e-01 1.96189079783958726654891080443122e-02
            3.09641800485391083253006172526511e-01 5.58961261915059681726347662333865e-01 1.31396937599549262776221780768537e-01 1.96189079783958726654891080443122e-02
            5.58961261915059681726347662333865e-01 1.31396937599549262776221780768537e-01 3.09641800485391083253006172526511e-01 1.96189079783958726654891080443122e-02
            5.58961261915059681726347662333865e-01 3.09641800485391083253006172526511e-01 1.31396937599549262776221780768537e-01 1.96189079783958726654891080443122e-02
            ];
        
    case  22
        
        % ALG. DEG.:   22
        % PTS CARD.:   96
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.7e-16
        
        xyw_bar=[
            7.22823551590805277872187062371268e-03 7.22823551590805277872187062371268e-03 9.85543528968183868421704119100468e-01 6.97145696250740581553784203094892e-04
            7.22823551590805277872187062371268e-03 9.85543528968183868421704119100468e-01 7.22823551590805277872187062371268e-03 6.97145696250740581553784203094892e-04
            9.85543528968183868421704119100468e-01 7.22823551590805277872187062371268e-03 7.22823551590805277872187062371268e-03 6.97145696250740581553784203094892e-04
            4.99714916562231881158595570013858e-01 4.99714916562231881158595570013858e-01 5.70166875536237682808859972283244e-04 1.73365376198036328508900183464903e-03
            4.99714916562231881158595570013858e-01 5.70166875536237682808859972283244e-04 4.99714916562231881158595570013858e-01 1.73365376198036328508900183464903e-03
            5.70166875536237682808859972283244e-04 4.99714916562231881158595570013858e-01 4.99714916562231881158595570013858e-01 1.73365376198036328508900183464903e-03
            4.19525272728040882208588868707011e-02 4.19525272728040882208588868707011e-02 9.16094945454391851313857841887511e-01 4.66131304304557517875773342552748e-03
            4.19525272728040882208588868707011e-02 9.16094945454391851313857841887511e-01 4.19525272728040882208588868707011e-02 4.66131304304557517875773342552748e-03
            9.16094945454391851313857841887511e-01 4.19525272728040882208588868707011e-02 4.19525272728040882208588868707011e-02 4.66131304304557517875773342552748e-03
            3.14132191520500925197012520584394e-01 3.14132191520500925197012520584394e-01 3.71735616958998149605974958831212e-01 1.15845767655995690592618529990432e-02
            3.14132191520500925197012520584394e-01 3.71735616958998149605974958831212e-01 3.14132191520500925197012520584394e-01 1.15845767655995690592618529990432e-02
            3.71735616958998149605974958831212e-01 3.14132191520500925197012520584394e-01 3.14132191520500925197012520584394e-01 1.15845767655995690592618529990432e-02
            4.80013651617116565706311348549207e-01 4.80013651617116565706311348549207e-01 3.99726967657668685873773029015865e-02 1.18789740330219155406421549514562e-02
            4.80013651617116565706311348549207e-01 3.99726967657668685873773029015865e-02 4.80013651617116565706311348549207e-01 1.18789740330219155406421549514562e-02
            3.99726967657668685873773029015865e-02 4.80013651617116565706311348549207e-01 4.80013651617116565706311348549207e-01 1.18789740330219155406421549514562e-02
            1.23087454230161152946010361119988e-01 1.23087454230161152946010361119988e-01 7.53825091539677694107979277760023e-01 1.53388036556667065046788422932877e-02
            1.23087454230161152946010361119988e-01 7.53825091539677694107979277760023e-01 1.23087454230161152946010361119988e-01 1.53388036556667065046788422932877e-02
            7.53825091539677694107979277760023e-01 1.23087454230161152946010361119988e-01 1.23087454230161152946010361119988e-01 1.53388036556667065046788422932877e-02
            4.45719630439115332976030003919732e-01 4.45719630439115332976030003919732e-01 1.08560739121769334047939992160536e-01 1.97163769402723468293636699399940e-02
            4.45719630439115332976030003919732e-01 1.08560739121769334047939992160536e-01 4.45719630439115332976030003919732e-01 1.97163769402723468293636699399940e-02
            1.08560739121769334047939992160536e-01 4.45719630439115332976030003919732e-01 4.45719630439115332976030003919732e-01 1.97163769402723468293636699399940e-02
            1.89862412183893686234981146299106e-01 1.89862412183893686234981146299106e-01 6.20275175632212683041188938659616e-01 2.05721514917347643447431693175531e-02
            1.89862412183893686234981146299106e-01 6.20275175632212683041188938659616e-01 1.89862412183893686234981146299106e-01 2.05721514917347643447431693175531e-02
            6.20275175632212683041188938659616e-01 1.89862412183893686234981146299106e-01 1.89862412183893686234981146299106e-01 2.05721514917347643447431693175531e-02
            2.60339233381755796870038466295227e-01 2.60339233381755796870038466295227e-01 4.79321533236488406259923067409545e-01 2.38406602781370512145198148346026e-02
            2.60339233381755796870038466295227e-01 4.79321533236488406259923067409545e-01 2.60339233381755796870038466295227e-01 2.38406602781370512145198148346026e-02
            4.79321533236488406259923067409545e-01 2.60339233381755796870038466295227e-01 2.60339233381755796870038466295227e-01 2.38406602781370512145198148346026e-02
            3.96366847671625999183220301347319e-01 3.96366847671625999183220301347319e-01 2.07266304656748001633559397305362e-01 2.44611970898748823377566452563769e-02
            3.96366847671625999183220301347319e-01 2.07266304656748001633559397305362e-01 3.96366847671625999183220301347319e-01 2.44611970898748823377566452563769e-02
            2.07266304656748001633559397305362e-01 3.96366847671625999183220301347319e-01 3.96366847671625999183220301347319e-01 2.44611970898748823377566452563769e-02
            7.73958468860307807757337172915868e-03 3.91001678273283775899571423906309e-02 9.53160247484068579026939005416352e-01 1.82862802731121004522207851294979e-03
            7.73958468860307807757337172915868e-03 9.53160247484068579026939005416352e-01 3.91001678273283775899571423906309e-02 1.82862802731121004522207851294979e-03
            3.91001678273283775899571423906309e-02 7.73958468860307807757337172915868e-03 9.53160247484068579026939005416352e-01 1.82862802731121004522207851294979e-03
            3.91001678273283775899571423906309e-02 9.53160247484068579026939005416352e-01 7.73958468860307807757337172915868e-03 1.82862802731121004522207851294979e-03
            9.53160247484068579026939005416352e-01 7.73958468860307807757337172915868e-03 3.91001678273283775899571423906309e-02 1.82862802731121004522207851294979e-03
            9.53160247484068579026939005416352e-01 3.91001678273283775899571423906309e-02 7.73958468860307807757337172915868e-03 1.82862802731121004522207851294979e-03
            5.77064130218403839069074567902362e-04 2.99563010316006073097128137305845e-01 6.99859925553775519269095184426988e-01 1.88044970396283915928248298143899e-03
            5.77064130218403839069074567902362e-04 6.99859925553775519269095184426988e-01 2.99563010316006073097128137305845e-01 1.88044970396283915928248298143899e-03
            2.99563010316006073097128137305845e-01 5.77064130218403839069074567902362e-04 6.99859925553775519269095184426988e-01 1.88044970396283915928248298143899e-03
            2.99563010316006073097128137305845e-01 6.99859925553775519269095184426988e-01 5.77064130218403839069074567902362e-04 1.88044970396283915928248298143899e-03
            6.99859925553775519269095184426988e-01 5.77064130218403839069074567902362e-04 2.99563010316006073097128137305845e-01 1.88044970396283915928248298143899e-03
            6.99859925553775519269095184426988e-01 2.99563010316006073097128137305845e-01 5.77064130218403839069074567902362e-04 1.88044970396283915928248298143899e-03
            9.79577185352314104560544905098141e-03 9.91675808522524682153687081154203e-02 8.91036647294224382065408462949563e-01 3.70410864001110380280823441978555e-03
            9.79577185352314104560544905098141e-03 8.91036647294224382065408462949563e-01 9.91675808522524682153687081154203e-02 3.70410864001110380280823441978555e-03
            9.91675808522524682153687081154203e-02 9.79577185352314104560544905098141e-03 8.91036647294224382065408462949563e-01 3.70410864001110380280823441978555e-03
            9.91675808522524682153687081154203e-02 8.91036647294224382065408462949563e-01 9.79577185352314104560544905098141e-03 3.70410864001110380280823441978555e-03
            8.91036647294224382065408462949563e-01 9.79577185352314104560544905098141e-03 9.91675808522524682153687081154203e-02 3.70410864001110380280823441978555e-03
            8.91036647294224382065408462949563e-01 9.91675808522524682153687081154203e-02 9.79577185352314104560544905098141e-03 3.70410864001110380280823441978555e-03
            9.18868020649896491414132526642788e-03 1.86190681552805714638054723764071e-01 8.04620638240695296161675287294202e-01 4.66834037183480772548183068693106e-03
            9.18868020649896491414132526642788e-03 8.04620638240695296161675287294202e-01 1.86190681552805714638054723764071e-01 4.66834037183480772548183068693106e-03
            1.86190681552805714638054723764071e-01 9.18868020649896491414132526642788e-03 8.04620638240695296161675287294202e-01 4.66834037183480772548183068693106e-03
            1.86190681552805714638054723764071e-01 8.04620638240695296161675287294202e-01 9.18868020649896491414132526642788e-03 4.66834037183480772548183068693106e-03
            8.04620638240695296161675287294202e-01 9.18868020649896491414132526642788e-03 1.86190681552805714638054723764071e-01 4.66834037183480772548183068693106e-03
            8.04620638240695296161675287294202e-01 1.86190681552805714638054723764071e-01 9.18868020649896491414132526642788e-03 4.66834037183480772548183068693106e-03
            1.34670875684018177820266970456942e-02 4.02938891855100456140803544258233e-01 5.83594020576497674035465479391860e-01 7.21134117192958907366362453217334e-03
            1.34670875684018177820266970456942e-02 5.83594020576497674035465479391860e-01 4.02938891855100456140803544258233e-01 7.21134117192958907366362453217334e-03
            4.02938891855100456140803544258233e-01 1.34670875684018177820266970456942e-02 5.83594020576497674035465479391860e-01 7.21134117192958907366362453217334e-03
            4.02938891855100456140803544258233e-01 5.83594020576497674035465479391860e-01 1.34670875684018177820266970456942e-02 7.21134117192958907366362453217334e-03
            5.83594020576497674035465479391860e-01 1.34670875684018177820266970456942e-02 4.02938891855100456140803544258233e-01 7.21134117192958907366362453217334e-03
            5.83594020576497674035465479391860e-01 4.02938891855100456140803544258233e-01 1.34670875684018177820266970456942e-02 7.21134117192958907366362453217334e-03
            5.06263124673368980488419310859172e-02 1.03588085409291874583459502900951e-01 8.45785602123371171856547334755305e-01 7.96743335988734681374356938476922e-03
            5.06263124673368980488419310859172e-02 8.45785602123371171856547334755305e-01 1.03588085409291874583459502900951e-01 7.96743335988734681374356938476922e-03
            1.03588085409291874583459502900951e-01 5.06263124673368980488419310859172e-02 8.45785602123371171856547334755305e-01 7.96743335988734681374356938476922e-03
            1.03588085409291874583459502900951e-01 8.45785602123371171856547334755305e-01 5.06263124673368980488419310859172e-02 7.96743335988734681374356938476922e-03
            8.45785602123371171856547334755305e-01 5.06263124673368980488419310859172e-02 1.03588085409291874583459502900951e-01 7.96743335988734681374356938476922e-03
            8.45785602123371171856547334755305e-01 1.03588085409291874583459502900951e-01 5.06263124673368980488419310859172e-02 7.96743335988734681374356938476922e-03
            4.83881177618376498439367594528449e-02 1.80823006888328891017536648178066e-01 7.70788875349833535466359535348602e-01 9.03580036943713894859175894680448e-03
            4.83881177618376498439367594528449e-02 7.70788875349833535466359535348602e-01 1.80823006888328891017536648178066e-01 9.03580036943713894859175894680448e-03
            1.80823006888328891017536648178066e-01 4.83881177618376498439367594528449e-02 7.70788875349833535466359535348602e-01 9.03580036943713894859175894680448e-03
            1.80823006888328891017536648178066e-01 7.70788875349833535466359535348602e-01 4.83881177618376498439367594528449e-02 9.03580036943713894859175894680448e-03
            7.70788875349833535466359535348602e-01 4.83881177618376498439367594528449e-02 1.80823006888328891017536648178066e-01 9.03580036943713894859175894680448e-03
            7.70788875349833535466359535348602e-01 1.80823006888328891017536648178066e-01 4.83881177618376498439367594528449e-02 9.03580036943713894859175894680448e-03
            2.74914358532603458895593462329998e-02 2.81943415079491888075580163786071e-01 6.90565149067247752157072682166472e-01 9.48064111155901868832618362148423e-03
            2.74914358532603458895593462329998e-02 6.90565149067247752157072682166472e-01 2.81943415079491888075580163786071e-01 9.48064111155901868832618362148423e-03
            2.81943415079491888075580163786071e-01 2.74914358532603458895593462329998e-02 6.90565149067247752157072682166472e-01 9.48064111155901868832618362148423e-03
            2.81943415079491888075580163786071e-01 6.90565149067247752157072682166472e-01 2.74914358532603458895593462329998e-02 9.48064111155901868832618362148423e-03
            6.90565149067247752157072682166472e-01 2.74914358532603458895593462329998e-02 2.81943415079491888075580163786071e-01 9.48064111155901868832618362148423e-03
            6.90565149067247752157072682166472e-01 2.81943415079491888075580163786071e-01 2.74914358532603458895593462329998e-02 9.48064111155901868832618362148423e-03
            6.71339744477043864900878133994411e-02 3.50712853397991353165963346327771e-01 5.82153172154304288099524455901701e-01 1.56173701283532154193700591804372e-02
            6.71339744477043864900878133994411e-02 5.82153172154304288099524455901701e-01 3.50712853397991353165963346327771e-01 1.56173701283532154193700591804372e-02
            3.50712853397991353165963346327771e-01 6.71339744477043864900878133994411e-02 5.82153172154304288099524455901701e-01 1.56173701283532154193700591804372e-02
            3.50712853397991353165963346327771e-01 5.82153172154304288099524455901701e-01 6.71339744477043864900878133994411e-02 1.56173701283532154193700591804372e-02
            5.82153172154304288099524455901701e-01 6.71339744477043864900878133994411e-02 3.50712853397991353165963346327771e-01 1.56173701283532154193700591804372e-02
            5.82153172154304288099524455901701e-01 3.50712853397991353165963346327771e-01 6.71339744477043864900878133994411e-02 1.56173701283532154193700591804372e-02
            9.67043036168603459268311439700483e-02 2.30176404403601320147743081179215e-01 6.73119291979538347803213582665194e-01 1.62453191810569899655725123466254e-02
            9.67043036168603459268311439700483e-02 6.73119291979538347803213582665194e-01 2.30176404403601320147743081179215e-01 1.62453191810569899655725123466254e-02
            2.30176404403601320147743081179215e-01 9.67043036168603459268311439700483e-02 6.73119291979538347803213582665194e-01 1.62453191810569899655725123466254e-02
            2.30176404403601320147743081179215e-01 6.73119291979538347803213582665194e-01 9.67043036168603459268311439700483e-02 1.62453191810569899655725123466254e-02
            6.73119291979538347803213582665194e-01 9.67043036168603459268311439700483e-02 2.30176404403601320147743081179215e-01 1.62453191810569899655725123466254e-02
            6.73119291979538347803213582665194e-01 2.30176404403601320147743081179215e-01 9.67043036168603459268311439700483e-02 1.62453191810569899655725123466254e-02
            1.51930454826702282522532527764270e-01 3.11593235863486472947414540612954e-01 5.36476309309811272285628547251690e-01 2.17848082235315224086669161351892e-02
            1.51930454826702282522532527764270e-01 5.36476309309811272285628547251690e-01 3.11593235863486472947414540612954e-01 2.17848082235315224086669161351892e-02
            3.11593235863486472947414540612954e-01 1.51930454826702282522532527764270e-01 5.36476309309811272285628547251690e-01 2.17848082235315224086669161351892e-02
            3.11593235863486472947414540612954e-01 5.36476309309811272285628547251690e-01 1.51930454826702282522532527764270e-01 2.17848082235315224086669161351892e-02
            5.36476309309811272285628547251690e-01 1.51930454826702282522532527764270e-01 3.11593235863486472947414540612954e-01 2.17848082235315224086669161351892e-02
            5.36476309309811272285628547251690e-01 3.11593235863486472947414540612954e-01 1.51930454826702282522532527764270e-01 2.17848082235315224086669161351892e-02
            ];
        
    case  24
        
        % ALG. DEG.:   24
        % PTS CARD.:  112
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.3e-15
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 2.29379434809674868922524382242045e-02
            7.64481587209140354965963837230447e-03 7.64481587209140354965963837230447e-03 9.84710368255817147797870347858407e-01 7.67377461115814011455293908170461e-04
            7.64481587209140354965963837230447e-03 9.84710368255817147797870347858407e-01 7.64481587209140354965963837230447e-03 7.67377461115814011455293908170461e-04
            9.84710368255817147797870347858407e-01 7.64481587209140354965963837230447e-03 7.64481587209140354965963837230447e-03 7.67377461115814011455293908170461e-04
            4.95123266934894168400660419138148e-01 4.95123266934894168400660419138148e-01 9.75346613021166319867916172370315e-03 5.52357844475169675046988260191938e-03
            4.95123266934894168400660419138148e-01 9.75346613021166319867916172370315e-03 4.95123266934894168400660419138148e-01 5.52357844475169675046988260191938e-03
            9.75346613021166319867916172370315e-03 4.95123266934894168400660419138148e-01 4.95123266934894168400660419138148e-01 5.52357844475169675046988260191938e-03
            8.53357147077405536750305259374727e-02 8.53357147077405536750305259374727e-02 8.29328570584518920405514563753968e-01 8.38499530275630886033688682346110e-03
            8.53357147077405536750305259374727e-02 8.29328570584518920405514563753968e-01 8.53357147077405536750305259374727e-02 8.38499530275630886033688682346110e-03
            8.29328570584518920405514563753968e-01 8.53357147077405536750305259374727e-02 8.53357147077405536750305259374727e-02 8.38499530275630886033688682346110e-03
            4.75404357579430003610809762903955e-01 4.75404357579430003610809762903955e-01 4.91912848411399927783804741920903e-02 1.22173801780023488783299967508356e-02
            4.75404357579430003610809762903955e-01 4.91912848411399927783804741920903e-02 4.75404357579430003610809762903955e-01 1.22173801780023488783299967508356e-02
            4.91912848411399927783804741920903e-02 4.75404357579430003610809762903955e-01 4.75404357579430003610809762903955e-01 1.22173801780023488783299967508356e-02
            1.43613126002549662540985764280776e-01 1.43613126002549662540985764280776e-01 7.12773747994900674918028471438447e-01 1.30703906056117789141746143855016e-02
            1.43613126002549662540985764280776e-01 7.12773747994900674918028471438447e-01 1.43613126002549662540985764280776e-01 1.30703906056117789141746143855016e-02
            7.12773747994900674918028471438447e-01 1.43613126002549662540985764280776e-01 1.43613126002549662540985764280776e-01 1.30703906056117789141746143855016e-02
            4.40001375500573721488706269155955e-01 4.40001375500573721488706269155955e-01 1.19997248998852557022587461688090e-01 1.78812277584295756249765929624118e-02
            4.40001375500573721488706269155955e-01 1.19997248998852557022587461688090e-01 4.40001375500573721488706269155955e-01 1.78812277584295756249765929624118e-02
            1.19997248998852557022587461688090e-01 4.40001375500573721488706269155955e-01 4.40001375500573721488706269155955e-01 1.78812277584295756249765929624118e-02
            2.04386847871303928281250250620360e-01 2.04386847871303928281250250620360e-01 5.91226304257392198948650730017107e-01 1.82217441390973046433376936192872e-02
            2.04386847871303928281250250620360e-01 5.91226304257392198948650730017107e-01 2.04386847871303928281250250620360e-01 1.82217441390973046433376936192872e-02
            5.91226304257392198948650730017107e-01 2.04386847871303928281250250620360e-01 2.04386847871303928281250250620360e-01 1.82217441390973046433376936192872e-02
            3.91555207032681984458122315118089e-01 3.91555207032681984458122315118089e-01 2.16889585934636031083755369763821e-01 2.16414972175676123544718620905769e-02
            3.91555207032681984458122315118089e-01 2.16889585934636031083755369763821e-01 3.91555207032681984458122315118089e-01 2.16414972175676123544718620905769e-02
            2.16889585934636031083755369763821e-01 3.91555207032681984458122315118089e-01 3.91555207032681984458122315118089e-01 2.16414972175676123544718620905769e-02
            2.69360691849568234523815135617042e-01 2.69360691849568234523815135617042e-01 4.61278616300863530952369728765916e-01 2.16883129269225989610436755583578e-02
            2.69360691849568234523815135617042e-01 4.61278616300863530952369728765916e-01 2.69360691849568234523815135617042e-01 2.16883129269225989610436755583578e-02
            4.61278616300863530952369728765916e-01 2.69360691849568234523815135617042e-01 2.69360691849568234523815135617042e-01 2.16883129269225989610436755583578e-02
            1.28915765322182144549112425835347e-03 3.89300321173704000354831578079029e-01 6.09410521173074259948521103069652e-01 1.74618458266172118902026522135884e-03
            1.28915765322182144549112425835347e-03 6.09410521173074259948521103069652e-01 3.89300321173704000354831578079029e-01 1.74618458266172118902026522135884e-03
            3.89300321173704000354831578079029e-01 1.28915765322182144549112425835347e-03 6.09410521173074259948521103069652e-01 1.74618458266172118902026522135884e-03
            3.89300321173704000354831578079029e-01 6.09410521173074259948521103069652e-01 1.28915765322182144549112425835347e-03 1.74618458266172118902026522135884e-03
            6.09410521173074259948521103069652e-01 1.28915765322182144549112425835347e-03 3.89300321173704000354831578079029e-01 1.74618458266172118902026522135884e-03
            6.09410521173074259948521103069652e-01 3.89300321173704000354831578079029e-01 1.28915765322182144549112425835347e-03 1.74618458266172118902026522135884e-03
            8.04615544134750990123983171997679e-03 3.99455189985108982675399147410644e-02 9.52008325560141699384075764100999e-01 1.83857160374278688283466998143467e-03
            8.04615544134750990123983171997679e-03 9.52008325560141699384075764100999e-01 3.99455189985108982675399147410644e-02 1.83857160374278688283466998143467e-03
            3.99455189985108982675399147410644e-02 8.04615544134750990123983171997679e-03 9.52008325560141699384075764100999e-01 1.83857160374278688283466998143467e-03
            3.99455189985108982675399147410644e-02 9.52008325560141699384075764100999e-01 8.04615544134750990123983171997679e-03 1.83857160374278688283466998143467e-03
            9.52008325560141699384075764100999e-01 8.04615544134750990123983171997679e-03 3.99455189985108982675399147410644e-02 1.83857160374278688283466998143467e-03
            9.52008325560141699384075764100999e-01 3.99455189985108982675399147410644e-02 8.04615544134750990123983171997679e-03 1.83857160374278688283466998143467e-03
            3.97463904079080419112779409829272e-03 1.77567553518447190930018564358761e-01 8.18457807440762019624003187345807e-01 1.99893663336212857109530638410888e-03
            3.97463904079080419112779409829272e-03 8.18457807440762019624003187345807e-01 1.77567553518447190930018564358761e-01 1.99893663336212857109530638410888e-03
            1.77567553518447190930018564358761e-01 3.97463904079080419112779409829272e-03 8.18457807440762019624003187345807e-01 1.99893663336212857109530638410888e-03
            1.77567553518447190930018564358761e-01 8.18457807440762019624003187345807e-01 3.97463904079080419112779409829272e-03 1.99893663336212857109530638410888e-03
            8.18457807440762019624003187345807e-01 3.97463904079080419112779409829272e-03 1.77567553518447190930018564358761e-01 1.99893663336212857109530638410888e-03
            8.18457807440762019624003187345807e-01 1.77567553518447190930018564358761e-01 3.97463904079080419112779409829272e-03 1.99893663336212857109530638410888e-03
            5.65339683877998013666887899830726e-03 9.73043731633908004496902321989182e-02 8.97042229997829232424066958628828e-01 2.06369617133590754765037900142488e-03
            5.65339683877998013666887899830726e-03 8.97042229997829232424066958628828e-01 9.73043731633908004496902321989182e-02 2.06369617133590754765037900142488e-03
            9.73043731633908004496902321989182e-02 5.65339683877998013666887899830726e-03 8.97042229997829232424066958628828e-01 2.06369617133590754765037900142488e-03
            9.73043731633908004496902321989182e-02 8.97042229997829232424066958628828e-01 5.65339683877998013666887899830726e-03 2.06369617133590754765037900142488e-03
            8.97042229997829232424066958628828e-01 5.65339683877998013666887899830726e-03 9.73043731633908004496902321989182e-02 2.06369617133590754765037900142488e-03
            8.97042229997829232424066958628828e-01 9.73043731633908004496902321989182e-02 5.65339683877998013666887899830726e-03 2.06369617133590754765037900142488e-03
            3.65416463470007168989894807964447e-02 4.73791364695558270270225875719916e-02 9.16079217183443428318412316002650e-01 2.29539508715841545499758424853098e-03
            3.65416463470007168989894807964447e-02 9.16079217183443428318412316002650e-01 4.73791364695558270270225875719916e-02 2.29539508715841545499758424853098e-03
            4.73791364695558270270225875719916e-02 3.65416463470007168989894807964447e-02 9.16079217183443428318412316002650e-01 2.29539508715841545499758424853098e-03
            4.73791364695558270270225875719916e-02 9.16079217183443428318412316002650e-01 3.65416463470007168989894807964447e-02 2.29539508715841545499758424853098e-03
            9.16079217183443428318412316002650e-01 3.65416463470007168989894807964447e-02 4.73791364695558270270225875719916e-02 2.29539508715841545499758424853098e-03
            9.16079217183443428318412316002650e-01 4.73791364695558270270225875719916e-02 3.65416463470007168989894807964447e-02 2.29539508715841545499758424853098e-03
            8.47203125278924405738845138102988e-03 2.76542334906226316970645484616398e-01 7.14985633840984347031621837231796e-01 4.57952328890823094631468137549746e-03
            8.47203125278924405738845138102988e-03 7.14985633840984347031621837231796e-01 2.76542334906226316970645484616398e-01 4.57952328890823094631468137549746e-03
            2.76542334906226316970645484616398e-01 8.47203125278924405738845138102988e-03 7.14985633840984347031621837231796e-01 4.57952328890823094631468137549746e-03
            2.76542334906226316970645484616398e-01 7.14985633840984347031621837231796e-01 8.47203125278924405738845138102988e-03 4.57952328890823094631468137549746e-03
            7.14985633840984347031621837231796e-01 8.47203125278924405738845138102988e-03 2.76542334906226316970645484616398e-01 4.57952328890823094631468137549746e-03
            7.14985633840984347031621837231796e-01 2.76542334906226316970645484616398e-01 8.47203125278924405738845138102988e-03 4.57952328890823094631468137549746e-03
            3.18687855584848661605334996238525e-02 1.01860048008413445197462010582967e-01 8.66271166433101758030943528865464e-01 5.29323009909540235368163862972324e-03
            3.18687855584848661605334996238525e-02 8.66271166433101758030943528865464e-01 1.01860048008413445197462010582967e-01 5.29323009909540235368163862972324e-03
            1.01860048008413445197462010582967e-01 3.18687855584848661605334996238525e-02 8.66271166433101758030943528865464e-01 5.29323009909540235368163862972324e-03
            1.01860048008413445197462010582967e-01 8.66271166433101758030943528865464e-01 3.18687855584848661605334996238525e-02 5.29323009909540235368163862972324e-03
            8.66271166433101758030943528865464e-01 3.18687855584848661605334996238525e-02 1.01860048008413445197462010582967e-01 5.29323009909540235368163862972324e-03
            8.66271166433101758030943528865464e-01 1.01860048008413445197462010582967e-01 3.18687855584848661605334996238525e-02 5.29323009909540235368163862972324e-03
            2.50744857569593221358061896353320e-02 1.79900411429897849613368521204393e-01 7.95025102813142758861886250087991e-01 5.57874699599088079426945085970146e-03
            2.50744857569593221358061896353320e-02 7.95025102813142758861886250087991e-01 1.79900411429897849613368521204393e-01 5.57874699599088079426945085970146e-03
            1.79900411429897849613368521204393e-01 2.50744857569593221358061896353320e-02 7.95025102813142758861886250087991e-01 5.57874699599088079426945085970146e-03
            1.79900411429897849613368521204393e-01 7.95025102813142758861886250087991e-01 2.50744857569593221358061896353320e-02 5.57874699599088079426945085970146e-03
            7.95025102813142758861886250087991e-01 2.50744857569593221358061896353320e-02 1.79900411429897849613368521204393e-01 5.57874699599088079426945085970146e-03
            7.95025102813142758861886250087991e-01 1.79900411429897849613368521204393e-01 2.50744857569593221358061896353320e-02 5.57874699599088079426945085970146e-03
            2.49546492072369834047673720078819e-02 3.74761488291451505538987021282082e-01 6.00283862501311493709010846941965e-01 8.61933948725663971746069336177243e-03
            2.49546492072369834047673720078819e-02 6.00283862501311493709010846941965e-01 3.74761488291451505538987021282082e-01 8.61933948725663971746069336177243e-03
            3.74761488291451505538987021282082e-01 2.49546492072369834047673720078819e-02 6.00283862501311493709010846941965e-01 8.61933948725663971746069336177243e-03
            3.74761488291451505538987021282082e-01 6.00283862501311493709010846941965e-01 2.49546492072369834047673720078819e-02 8.61933948725663971746069336177243e-03
            6.00283862501311493709010846941965e-01 2.49546492072369834047673720078819e-02 3.74761488291451505538987021282082e-01 8.61933948725663971746069336177243e-03
            6.00283862501311493709010846941965e-01 3.74761488291451505538987021282082e-01 2.49546492072369834047673720078819e-02 8.61933948725663971746069336177243e-03
            7.18069274899266751788218243746087e-02 1.62853713656418902511191504345334e-01 7.65339358853654450065562286908971e-01 9.99235399224706818954722109538125e-03
            7.18069274899266751788218243746087e-02 7.65339358853654450065562286908971e-01 1.62853713656418902511191504345334e-01 9.99235399224706818954722109538125e-03
            1.62853713656418902511191504345334e-01 7.18069274899266751788218243746087e-02 7.65339358853654450065562286908971e-01 9.99235399224706818954722109538125e-03
            1.62853713656418902511191504345334e-01 7.65339358853654450065562286908971e-01 7.18069274899266751788218243746087e-02 9.99235399224706818954722109538125e-03
            7.65339358853654450065562286908971e-01 7.18069274899266751788218243746087e-02 1.62853713656418902511191504345334e-01 9.99235399224706818954722109538125e-03
            7.65339358853654450065562286908971e-01 1.62853713656418902511191504345334e-01 7.18069274899266751788218243746087e-02 9.99235399224706818954722109538125e-03
            4.64953683252256055880025087390095e-02 2.62999946925326910651676826091716e-01 6.90504684749447483760320665169274e-01 1.03056162076813062622449024274829e-02
            4.64953683252256055880025087390095e-02 6.90504684749447483760320665169274e-01 2.62999946925326910651676826091716e-01 1.03056162076813062622449024274829e-02
            2.62999946925326910651676826091716e-01 4.64953683252256055880025087390095e-02 6.90504684749447483760320665169274e-01 1.03056162076813062622449024274829e-02
            2.62999946925326910651676826091716e-01 6.90504684749447483760320665169274e-01 4.64953683252256055880025087390095e-02 1.03056162076813062622449024274829e-02
            6.90504684749447483760320665169274e-01 4.64953683252256055880025087390095e-02 2.62999946925326910651676826091716e-01 1.03056162076813062622449024274829e-02
            6.90504684749447483760320665169274e-01 2.62999946925326910651676826091716e-01 4.64953683252256055880025087390095e-02 1.03056162076813062622449024274829e-02
            7.94456583289277762371938251817483e-02 3.53874622686032425100677301088581e-01 5.66679718985039770906553258100757e-01 1.46045461262502441340815906301032e-02
            7.94456583289277762371938251817483e-02 5.66679718985039770906553258100757e-01 3.53874622686032425100677301088581e-01 1.46045461262502441340815906301032e-02
            3.53874622686032425100677301088581e-01 7.94456583289277762371938251817483e-02 5.66679718985039770906553258100757e-01 1.46045461262502441340815906301032e-02
            3.53874622686032425100677301088581e-01 5.66679718985039770906553258100757e-01 7.94456583289277762371938251817483e-02 1.46045461262502441340815906301032e-02
            5.66679718985039770906553258100757e-01 7.94456583289277762371938251817483e-02 3.53874622686032425100677301088581e-01 1.46045461262502441340815906301032e-02
            5.66679718985039770906553258100757e-01 3.53874622686032425100677301088581e-01 7.94456583289277762371938251817483e-02 1.46045461262502441340815906301032e-02
            1.12104968398914525695708732655476e-01 2.40452720011096854335619354969822e-01 6.47442311589988661602035335818073e-01 1.49985006762504598232244035216354e-02
            1.12104968398914525695708732655476e-01 6.47442311589988661602035335818073e-01 2.40452720011096854335619354969822e-01 1.49985006762504598232244035216354e-02
            2.40452720011096854335619354969822e-01 1.12104968398914525695708732655476e-01 6.47442311589988661602035335818073e-01 1.49985006762504598232244035216354e-02
            2.40452720011096854335619354969822e-01 6.47442311589988661602035335818073e-01 1.12104968398914525695708732655476e-01 1.49985006762504598232244035216354e-02
            6.47442311589988661602035335818073e-01 1.12104968398914525695708732655476e-01 2.40452720011096854335619354969822e-01 1.49985006762504598232244035216354e-02
            6.47442311589988661602035335818073e-01 2.40452720011096854335619354969822e-01 1.12104968398914525695708732655476e-01 1.49985006762504598232244035216354e-02
            1.62887044320893126236882153534680e-01 3.16328836361366161344932379506645e-01 5.20784119317740712418185466958676e-01 1.92307831174366762605476566250218e-02
            1.62887044320893126236882153534680e-01 5.20784119317740712418185466958676e-01 3.16328836361366161344932379506645e-01 1.92307831174366762605476566250218e-02
            3.16328836361366161344932379506645e-01 1.62887044320893126236882153534680e-01 5.20784119317740712418185466958676e-01 1.92307831174366762605476566250218e-02
            3.16328836361366161344932379506645e-01 5.20784119317740712418185466958676e-01 1.62887044320893126236882153534680e-01 1.92307831174366762605476566250218e-02
            5.20784119317740712418185466958676e-01 1.62887044320893126236882153534680e-01 3.16328836361366161344932379506645e-01 1.92307831174366762605476566250218e-02
            5.20784119317740712418185466958676e-01 3.16328836361366161344932379506645e-01 1.62887044320893126236882153534680e-01 1.92307831174366762605476566250218e-02
            ];
        
    case  25
        
        % ALG. DEG.:   25
        % PTS CARD.:  126
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.2e-16
        
        xyw_bar=[
            4.86026758463412866806407919284538e-01 4.86026758463412866806407919284538e-01 2.79464830731742663871841614309233e-02 8.00558188002041884456794207380881e-03
            4.86026758463412866806407919284538e-01 2.79464830731742663871841614309233e-02 4.86026758463412866806407919284538e-01 8.00558188002041884456794207380881e-03
            2.79464830731742663871841614309233e-02 4.86026758463412866806407919284538e-01 4.86026758463412866806407919284538e-01 8.00558188002041884456794207380881e-03
            4.34410699336174210838379394772346e-01 4.34410699336174210838379394772346e-01 1.31178601327651578323241210455308e-01 1.59470768323904839303128966321310e-02
            4.34410699336174210838379394772346e-01 1.31178601327651578323241210455308e-01 4.34410699336174210838379394772346e-01 1.59470768323904839303128966321310e-02
            1.31178601327651578323241210455308e-01 4.34410699336174210838379394772346e-01 4.34410699336174210838379394772346e-01 1.59470768323904839303128966321310e-02
            3.89889135243963846910730808303924e-01 3.89889135243963846910730808303924e-01 2.20221729512072306178538383392151e-01 1.31091412307954985339009112976782e-02
            3.89889135243963846910730808303924e-01 2.20221729512072306178538383392151e-01 3.89889135243963846910730808303924e-01 1.31091412307954985339009112976782e-02
            2.20221729512072306178538383392151e-01 3.89889135243963846910730808303924e-01 3.89889135243963846910730808303924e-01 1.31091412307954985339009112976782e-02
            2.98443234019804493062366645972361e-01 2.98443234019804493062366645972361e-01 4.03113531960391013875266708055278e-01 1.95830009656355955027784432331828e-02
            2.98443234019804493062366645972361e-01 4.03113531960391013875266708055278e-01 2.98443234019804493062366645972361e-01 1.95830009656355955027784432331828e-02
            4.03113531960391013875266708055278e-01 2.98443234019804493062366645972361e-01 2.98443234019804493062366645972361e-01 1.95830009656355955027784432331828e-02
            2.34044172337371847447684558574110e-01 2.34044172337371847447684558574110e-01 5.31911655325256305104630882851779e-01 1.64708854415372557555841126486484e-02
            2.34044172337371847447684558574110e-01 5.31911655325256305104630882851779e-01 2.34044172337371847447684558574110e-01 1.64708854415372557555841126486484e-02
            5.31911655325256305104630882851779e-01 2.34044172337371847447684558574110e-01 2.34044172337371847447684558574110e-01 1.64708854415372557555841126486484e-02
            1.51468334609017596381619341627811e-01 1.51468334609017596381619341627811e-01 6.97063330781964807236761316744378e-01 8.54727907409213831513294934438818e-03
            1.51468334609017596381619341627811e-01 6.97063330781964807236761316744378e-01 1.51468334609017596381619341627811e-01 8.54727907409213831513294934438818e-03
            6.97063330781964807236761316744378e-01 1.51468334609017596381619341627811e-01 1.51468334609017596381619341627811e-01 8.54727907409213831513294934438818e-03
            1.12733893545993488216616640329448e-01 1.12733893545993488216616640329448e-01 7.74532212908012995811191103712190e-01 8.16188585722649705134212894108714e-03
            1.12733893545993488216616640329448e-01 7.74532212908012995811191103712190e-01 1.12733893545993488216616640329448e-01 8.16188585722649705134212894108714e-03
            7.74532212908012995811191103712190e-01 1.12733893545993488216616640329448e-01 1.12733893545993488216616640329448e-01 8.16188585722649705134212894108714e-03
            7.77156920915262061644313007491292e-02 7.77156920915262061644313007491292e-02 8.44568615816947643182288629759569e-01 6.12114653998374787235992044998056e-03
            7.77156920915262061644313007491292e-02 8.44568615816947643182288629759569e-01 7.77156920915262061644313007491292e-02 6.12114653998374787235992044998056e-03
            8.44568615816947643182288629759569e-01 7.77156920915262061644313007491292e-02 7.77156920915262061644313007491292e-02 6.12114653998374787235992044998056e-03
            3.48930936142969050361983818220324e-02 3.48930936142969050361983818220324e-02 9.30213812771406134416452005098108e-01 2.90849826493665922916287414068393e-03
            3.48930936142969050361983818220324e-02 9.30213812771406134416452005098108e-01 3.48930936142969050361983818220324e-02 2.90849826493665922916287414068393e-03
            9.30213812771406134416452005098108e-01 3.48930936142969050361983818220324e-02 3.48930936142969050361983818220324e-02 2.90849826493665922916287414068393e-03
            7.25818462093238882942847212120796e-03 7.25818462093238882942847212120796e-03 9.85483630758135276117570811038604e-01 6.92275245661998456063457307863018e-04
            7.25818462093238882942847212120796e-03 9.85483630758135276117570811038604e-01 7.25818462093238882942847212120796e-03 6.92275245661998456063457307863018e-04
            9.85483630758135276117570811038604e-01 7.25818462093238882942847212120796e-03 7.25818462093238882942847212120796e-03 6.92275245661998456063457307863018e-04
            1.29235270444219584907996623002191e-03 2.27214452153364110387201435514726e-01 7.71493195142193721736134648381267e-01 1.24828919927740347207978377497284e-03
            1.29235270444219584907996623002191e-03 7.71493195142193721736134648381267e-01 2.27214452153364110387201435514726e-01 1.24828919927740347207978377497284e-03
            2.27214452153364110387201435514726e-01 1.29235270444219584907996623002191e-03 7.71493195142193721736134648381267e-01 1.24828919927740347207978377497284e-03
            2.27214452153364110387201435514726e-01 7.71493195142193721736134648381267e-01 1.29235270444219584907996623002191e-03 1.24828919927740347207978377497284e-03
            7.71493195142193721736134648381267e-01 1.29235270444219584907996623002191e-03 2.27214452153364110387201435514726e-01 1.24828919927740347207978377497284e-03
            7.71493195142193721736134648381267e-01 2.27214452153364110387201435514726e-01 1.29235270444219584907996623002191e-03 1.24828919927740347207978377497284e-03
            5.39970127211620177370976492170485e-03 4.35010554853571707045745142750093e-01 5.59589743874312040006202551012393e-01 3.40475290880302506116295013782747e-03
            5.39970127211620177370976492170485e-03 5.59589743874312040006202551012393e-01 4.35010554853571707045745142750093e-01 3.40475290880302506116295013782747e-03
            4.35010554853571707045745142750093e-01 5.39970127211620177370976492170485e-03 5.59589743874312040006202551012393e-01 3.40475290880302506116295013782747e-03
            4.35010554853571707045745142750093e-01 5.59589743874312040006202551012393e-01 5.39970127211620177370976492170485e-03 3.40475290880302506116295013782747e-03
            5.59589743874312040006202551012393e-01 5.39970127211620177370976492170485e-03 4.35010554853571707045745142750093e-01 3.40475290880302506116295013782747e-03
            5.59589743874312040006202551012393e-01 4.35010554853571707045745142750093e-01 5.39970127211620177370976492170485e-03 3.40475290880302506116295013782747e-03
            6.38400303397498956187172680643016e-03 3.20309599272204481579251478251535e-01 6.73306397693820501970662917301524e-01 3.35965432606405347421008045216695e-03
            6.38400303397498956187172680643016e-03 6.73306397693820501970662917301524e-01 3.20309599272204481579251478251535e-01 3.35965432606405347421008045216695e-03
            3.20309599272204481579251478251535e-01 6.38400303397498956187172680643016e-03 6.73306397693820501970662917301524e-01 3.35965432606405347421008045216695e-03
            3.20309599272204481579251478251535e-01 6.73306397693820501970662917301524e-01 6.38400303397498956187172680643016e-03 3.35965432606405347421008045216695e-03
            6.73306397693820501970662917301524e-01 6.38400303397498956187172680643016e-03 3.20309599272204481579251478251535e-01 3.35965432606405347421008045216695e-03
            6.73306397693820501970662917301524e-01 3.20309599272204481579251478251535e-01 6.38400303397498956187172680643016e-03 3.35965432606405347421008045216695e-03
            5.02821150199308283118559970148453e-03 9.17503222800053058749725209963799e-02 9.03221466218001589609798429592047e-01 1.71615653949675687865716877666955e-03
            5.02821150199308283118559970148453e-03 9.03221466218001589609798429592047e-01 9.17503222800053058749725209963799e-02 1.71615653949675687865716877666955e-03
            9.17503222800053058749725209963799e-02 5.02821150199308283118559970148453e-03 9.03221466218001589609798429592047e-01 1.71615653949675687865716877666955e-03
            9.17503222800053058749725209963799e-02 9.03221466218001589609798429592047e-01 5.02821150199308283118559970148453e-03 1.71615653949675687865716877666955e-03
            9.03221466218001589609798429592047e-01 5.02821150199308283118559970148453e-03 9.17503222800053058749725209963799e-02 1.71615653949675687865716877666955e-03
            9.03221466218001589609798429592047e-01 9.17503222800053058749725209963799e-02 5.02821150199308283118559970148453e-03 1.71615653949675687865716877666955e-03
            6.82675862178184610778997054580941e-03 3.80108358587244316018072254337312e-02 9.55162405519493717953594114078442e-01 1.48085631671560210773086474489446e-03
            6.82675862178184610778997054580941e-03 9.55162405519493717953594114078442e-01 3.80108358587244316018072254337312e-02 1.48085631671560210773086474489446e-03
            3.80108358587244316018072254337312e-02 6.82675862178184610778997054580941e-03 9.55162405519493717953594114078442e-01 1.48085631671560210773086474489446e-03
            3.80108358587244316018072254337312e-02 9.55162405519493717953594114078442e-01 6.82675862178184610778997054580941e-03 1.48085631671560210773086474489446e-03
            9.55162405519493717953594114078442e-01 6.82675862178184610778997054580941e-03 3.80108358587244316018072254337312e-02 1.48085631671560210773086474489446e-03
            9.55162405519493717953594114078442e-01 3.80108358587244316018072254337312e-02 6.82675862178184610778997054580941e-03 1.48085631671560210773086474489446e-03
            1.00161996399295120940831083089506e-02 1.57425218485311796401404649259348e-01 8.32558581874758729668428713921458e-01 3.51131261072867850714374071685597e-03
            1.00161996399295120940831083089506e-02 8.32558581874758729668428713921458e-01 1.57425218485311796401404649259348e-01 3.51131261072867850714374071685597e-03
            1.57425218485311796401404649259348e-01 1.00161996399295120940831083089506e-02 8.32558581874758729668428713921458e-01 3.51131261072867850714374071685597e-03
            1.57425218485311796401404649259348e-01 8.32558581874758729668428713921458e-01 1.00161996399295120940831083089506e-02 3.51131261072867850714374071685597e-03
            8.32558581874758729668428713921458e-01 1.00161996399295120940831083089506e-02 1.57425218485311796401404649259348e-01 3.51131261072867850714374071685597e-03
            8.32558581874758729668428713921458e-01 1.57425218485311796401404649259348e-01 1.00161996399295120940831083089506e-02 3.51131261072867850714374071685597e-03
            2.57578131733900463073982933792649e-02 2.39889659778533270717559844342759e-01 7.34352527048076630933337582973763e-01 7.39355014970648552474763448572048e-03
            2.57578131733900463073982933792649e-02 7.34352527048076630933337582973763e-01 2.39889659778533270717559844342759e-01 7.39355014970648552474763448572048e-03
            2.39889659778533270717559844342759e-01 2.57578131733900463073982933792649e-02 7.34352527048076630933337582973763e-01 7.39355014970648552474763448572048e-03
            2.39889659778533270717559844342759e-01 7.34352527048076630933337582973763e-01 2.57578131733900463073982933792649e-02 7.39355014970648552474763448572048e-03
            7.34352527048076630933337582973763e-01 2.57578131733900463073982933792649e-02 2.39889659778533270717559844342759e-01 7.39355014970648552474763448572048e-03
            7.34352527048076630933337582973763e-01 2.39889659778533270717559844342759e-01 2.57578131733900463073982933792649e-02 7.39355014970648552474763448572048e-03
            3.02278981199158233716950405778334e-02 3.61943118126060559003320804549730e-01 6.07828983754023610686090250965208e-01 7.98308747737656344833911958858153e-03
            3.02278981199158233716950405778334e-02 6.07828983754023610686090250965208e-01 3.61943118126060559003320804549730e-01 7.98308747737656344833911958858153e-03
            3.61943118126060559003320804549730e-01 3.02278981199158233716950405778334e-02 6.07828983754023610686090250965208e-01 7.98308747737656344833911958858153e-03
            3.61943118126060559003320804549730e-01 6.07828983754023610686090250965208e-01 3.02278981199158233716950405778334e-02 7.98308747737656344833911958858153e-03
            6.07828983754023610686090250965208e-01 3.02278981199158233716950405778334e-02 3.61943118126060559003320804549730e-01 7.98308747737656344833911958858153e-03
            6.07828983754023610686090250965208e-01 3.61943118126060559003320804549730e-01 3.02278981199158233716950405778334e-02 7.98308747737656344833911958858153e-03
            3.05049901071620770121040067124341e-02 8.35519609548284608591117716969165e-02 8.85943048938009458659337269637035e-01 4.35596261315803617852893125927949e-03
            3.05049901071620770121040067124341e-02 8.85943048938009458659337269637035e-01 8.35519609548284608591117716969165e-02 4.35596261315803617852893125927949e-03
            8.35519609548284608591117716969165e-02 3.05049901071620770121040067124341e-02 8.85943048938009458659337269637035e-01 4.35596261315803617852893125927949e-03
            8.35519609548284608591117716969165e-02 8.85943048938009458659337269637035e-01 3.05049901071620770121040067124341e-02 4.35596261315803617852893125927949e-03
            8.85943048938009458659337269637035e-01 3.05049901071620770121040067124341e-02 8.35519609548284608591117716969165e-02 4.35596261315803617852893125927949e-03
            8.85943048938009458659337269637035e-01 8.35519609548284608591117716969165e-02 3.05049901071620770121040067124341e-02 4.35596261315803617852893125927949e-03
            4.59565473625693221570998048264300e-02 1.48443220732418112994821512984345e-01 8.05600231905012509336927450931398e-01 7.36505670141782749910097649603813e-03
            4.59565473625693221570998048264300e-02 8.05600231905012509336927450931398e-01 1.48443220732418112994821512984345e-01 7.36505670141782749910097649603813e-03
            1.48443220732418112994821512984345e-01 4.59565473625693221570998048264300e-02 8.05600231905012509336927450931398e-01 7.36505670141782749910097649603813e-03
            1.48443220732418112994821512984345e-01 8.05600231905012509336927450931398e-01 4.59565473625693221570998048264300e-02 7.36505670141782749910097649603813e-03
            8.05600231905012509336927450931398e-01 4.59565473625693221570998048264300e-02 1.48443220732418112994821512984345e-01 7.36505670141782749910097649603813e-03
            8.05600231905012509336927450931398e-01 1.48443220732418112994821512984345e-01 4.59565473625693221570998048264300e-02 7.36505670141782749910097649603813e-03
            6.74428005402776115095875297811290e-02 2.83739708727534956089755269204034e-01 6.48817490732187418522869393200381e-01 1.09635728464195399134428754450710e-02
            6.74428005402776115095875297811290e-02 6.48817490732187418522869393200381e-01 2.83739708727534956089755269204034e-01 1.09635728464195399134428754450710e-02
            2.83739708727534956089755269204034e-01 6.74428005402776115095875297811290e-02 6.48817490732187418522869393200381e-01 1.09635728464195399134428754450710e-02
            2.83739708727534956089755269204034e-01 6.48817490732187418522869393200381e-01 6.74428005402776115095875297811290e-02 1.09635728464195399134428754450710e-02
            6.48817490732187418522869393200381e-01 6.74428005402776115095875297811290e-02 2.83739708727534956089755269204034e-01 1.09635728464195399134428754450710e-02
            6.48817490732187418522869393200381e-01 2.83739708727534956089755269204034e-01 6.74428005402776115095875297811290e-02 1.09635728464195399134428754450710e-02
            7.00450914159106152734324268749333e-02 4.06899375118787598459846321929945e-01 5.23055533465301758511145635566209e-01 1.17499617435411173038684040648150e-02
            7.00450914159106152734324268749333e-02 5.23055533465301758511145635566209e-01 4.06899375118787598459846321929945e-01 1.17499617435411173038684040648150e-02
            4.06899375118787598459846321929945e-01 7.00450914159106152734324268749333e-02 5.23055533465301758511145635566209e-01 1.17499617435411173038684040648150e-02
            4.06899375118787598459846321929945e-01 5.23055533465301758511145635566209e-01 7.00450914159106152734324268749333e-02 1.17499617435411173038684040648150e-02
            5.23055533465301758511145635566209e-01 7.00450914159106152734324268749333e-02 4.06899375118787598459846321929945e-01 1.17499617435411173038684040648150e-02
            5.23055533465301758511145635566209e-01 4.06899375118787598459846321929945e-01 7.00450914159106152734324268749333e-02 1.17499617435411173038684040648150e-02
            8.39115246401166414713657104584854e-02 1.94113987024892492572192281841126e-01 7.21974488334990893712017623329302e-01 1.00156007137985718358219600077064e-02
            8.39115246401166414713657104584854e-02 7.21974488334990893712017623329302e-01 1.94113987024892492572192281841126e-01 1.00156007137985718358219600077064e-02
            1.94113987024892492572192281841126e-01 8.39115246401166414713657104584854e-02 7.21974488334990893712017623329302e-01 1.00156007137985718358219600077064e-02
            1.94113987024892492572192281841126e-01 7.21974488334990893712017623329302e-01 8.39115246401166414713657104584854e-02 1.00156007137985718358219600077064e-02
            7.21974488334990893712017623329302e-01 8.39115246401166414713657104584854e-02 1.94113987024892492572192281841126e-01 1.00156007137985718358219600077064e-02
            7.21974488334990893712017623329302e-01 1.94113987024892492572192281841126e-01 8.39115246401166414713657104584854e-02 1.00156007137985718358219600077064e-02
            1.20375535677152700753111957965302e-01 3.24134347000703204599858509027399e-01 5.55490117322144039135878301749472e-01 1.33096407876286629973217046085665e-02
            1.20375535677152700753111957965302e-01 5.55490117322144039135878301749472e-01 3.24134347000703204599858509027399e-01 1.33096407876286629973217046085665e-02
            3.24134347000703204599858509027399e-01 1.20375535677152700753111957965302e-01 5.55490117322144039135878301749472e-01 1.33096407876286629973217046085665e-02
            3.24134347000703204599858509027399e-01 5.55490117322144039135878301749472e-01 1.20375535677152700753111957965302e-01 1.33096407876286629973217046085665e-02
            5.55490117322144039135878301749472e-01 1.20375535677152700753111957965302e-01 3.24134347000703204599858509027399e-01 1.33096407876286629973217046085665e-02
            5.55490117322144039135878301749472e-01 3.24134347000703204599858509027399e-01 1.20375535677152700753111957965302e-01 1.33096407876286629973217046085665e-02
            1.48066899157366704820049108093372e-01 2.29277483555981043128468854774837e-01 6.22655617286652196540330805873964e-01 1.41544465052261241727604712536959e-02
            1.48066899157366704820049108093372e-01 6.22655617286652196540330805873964e-01 2.29277483555981043128468854774837e-01 1.41544465052261241727604712536959e-02
            2.29277483555981043128468854774837e-01 1.48066899157366704820049108093372e-01 6.22655617286652196540330805873964e-01 1.41544465052261241727604712536959e-02
            2.29277483555981043128468854774837e-01 6.22655617286652196540330805873964e-01 1.48066899157366704820049108093372e-01 1.41544465052261241727604712536959e-02
            6.22655617286652196540330805873964e-01 1.48066899157366704820049108093372e-01 2.29277483555981043128468854774837e-01 1.41544465052261241727604712536959e-02
            6.22655617286652196540330805873964e-01 2.29277483555981043128468854774837e-01 1.48066899157366704820049108093372e-01 1.41544465052261241727604712536959e-02
            1.91771865867325114196617619199969e-01 3.25618122595983827682886158072506e-01 4.82610011536691085876071838356438e-01 1.48813795611679958869055795389613e-02
            1.91771865867325114196617619199969e-01 4.82610011536691085876071838356438e-01 3.25618122595983827682886158072506e-01 1.48813795611679958869055795389613e-02
            3.25618122595983827682886158072506e-01 1.91771865867325114196617619199969e-01 4.82610011536691085876071838356438e-01 1.48813795611679958869055795389613e-02
            3.25618122595983827682886158072506e-01 4.82610011536691085876071838356438e-01 1.91771865867325114196617619199969e-01 1.48813795611679958869055795389613e-02
            4.82610011536691085876071838356438e-01 1.91771865867325114196617619199969e-01 3.25618122595983827682886158072506e-01 1.48813795611679958869055795389613e-02
            4.82610011536691085876071838356438e-01 3.25618122595983827682886158072506e-01 1.91771865867325114196617619199969e-01 1.48813795611679958869055795389613e-02
            ];
        
    case  26
        
        % ALG. DEG.:   26
        % PTS CARD.:  133
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 8.5e-16
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 1.41775216099960278681724190619207e-02
            4.27380776384648330967763740773080e-01 4.27380776384648330967763740773080e-01 1.45238447230703338064472518453840e-01 1.56284773881481542368021564470837e-02
            4.27380776384648330967763740773080e-01 1.45238447230703338064472518453840e-01 4.27380776384648330967763740773080e-01 1.56284773881481542368021564470837e-02
            1.45238447230703338064472518453840e-01 4.27380776384648330967763740773080e-01 4.27380776384648330967763740773080e-01 1.56284773881481542368021564470837e-02
            3.86991728384621147540656238561496e-01 3.86991728384621147540656238561496e-01 2.26016543230757704918687522877008e-01 1.46375262291117864965617201278292e-02
            3.86991728384621147540656238561496e-01 2.26016543230757704918687522877008e-01 3.86991728384621147540656238561496e-01 1.46375262291117864965617201278292e-02
            2.26016543230757704918687522877008e-01 3.86991728384621147540656238561496e-01 3.86991728384621147540656238561496e-01 1.46375262291117864965617201278292e-02
            2.90801606496775999488590969122015e-01 2.90801606496775999488590969122015e-01 4.18396787006448001022818061755970e-01 1.19987734663434524734482167218630e-02
            2.90801606496775999488590969122015e-01 4.18396787006448001022818061755970e-01 2.90801606496775999488590969122015e-01 1.19987734663434524734482167218630e-02
            4.18396787006448001022818061755970e-01 2.90801606496775999488590969122015e-01 2.90801606496775999488590969122015e-01 1.19987734663434524734482167218630e-02
            2.19551193915611925211450738970598e-01 2.19551193915611925211450738970598e-01 5.60897612168776094065947290800977e-01 1.40892750357101432384654415841396e-02
            2.19551193915611925211450738970598e-01 5.60897612168776094065947290800977e-01 2.19551193915611925211450738970598e-01 1.40892750357101432384654415841396e-02
            5.60897612168776094065947290800977e-01 2.19551193915611925211450738970598e-01 2.19551193915611925211450738970598e-01 1.40892750357101432384654415841396e-02
            1.41477668767692438267502552662336e-01 1.41477668767692438267502552662336e-01 7.17044662464615178976146125933155e-01 1.13708950654174256039707913146231e-02
            1.41477668767692438267502552662336e-01 7.17044662464615178976146125933155e-01 1.41477668767692438267502552662336e-01 1.13708950654174256039707913146231e-02
            7.17044662464615178976146125933155e-01 1.41477668767692438267502552662336e-01 1.41477668767692438267502552662336e-01 1.13708950654174256039707913146231e-02
            8.52977390247050409888274202785396e-02 8.52977390247050409888274202785396e-02 8.29404521950589890266769543814007e-01 5.76148405319945166452866303075098e-03
            8.52977390247050409888274202785396e-02 8.29404521950589890266769543814007e-01 8.52977390247050409888274202785396e-02 5.76148405319945166452866303075098e-03
            8.29404521950589890266769543814007e-01 8.52977390247050409888274202785396e-02 8.52977390247050409888274202785396e-02 5.76148405319945166452866303075098e-03
            3.24537987119420390236257389915409e-02 3.24537987119420390236257389915409e-02 9.35092402576115921952748522016918e-01 2.88047484993807960743095186728624e-03
            3.24537987119420390236257389915409e-02 9.35092402576115921952748522016918e-01 3.24537987119420390236257389915409e-02 2.88047484993807960743095186728624e-03
            9.35092402576115921952748522016918e-01 3.24537987119420390236257389915409e-02 3.24537987119420390236257389915409e-02 2.88047484993807960743095186728624e-03
            6.88528565235726423138240548382782e-03 6.88528565235726423138240548382782e-03 9.86229428695285514905322088452522e-01 6.17924007806828698430301471944404e-04
            6.88528565235726423138240548382782e-03 9.86229428695285514905322088452522e-01 6.88528565235726423138240548382782e-03 6.17924007806828698430301471944404e-04
            9.86229428695285514905322088452522e-01 6.88528565235726423138240548382782e-03 6.88528565235726423138240548382782e-03 6.17924007806828698430301471944404e-04
            4.39945211557685067482736940291943e-01 5.53674216445258049468236549728317e-01 6.38057199705688304902650997973979e-03 3.69419331787614046991308214273886e-03
            4.39945211557685067482736940291943e-01 6.38057199705688304902650997973979e-03 5.53674216445258049468236549728317e-01 3.69419331787614046991308214273886e-03
            5.53674216445258049468236549728317e-01 4.39945211557685067482736940291943e-01 6.38057199705688304902650997973979e-03 3.69419331787614046991308214273886e-03
            5.53674216445258049468236549728317e-01 6.38057199705688304902650997973979e-03 4.39945211557685067482736940291943e-01 3.69419331787614046991308214273886e-03
            6.38057199705688304902650997973979e-03 4.39945211557685067482736940291943e-01 5.53674216445258049468236549728317e-01 3.69419331787614046991308214273886e-03
            6.38057199705688304902650997973979e-03 5.53674216445258049468236549728317e-01 4.39945211557685067482736940291943e-01 3.69419331787614046991308214273886e-03
            3.29997034833231284167709418397862e-01 6.63643196738745944074366889253724e-01 6.35976842802277175792369234841317e-03 3.43799094677205022488331032093356e-03
            3.29997034833231284167709418397862e-01 6.35976842802277175792369234841317e-03 6.63643196738745944074366889253724e-01 3.43799094677205022488331032093356e-03
            6.63643196738745944074366889253724e-01 3.29997034833231284167709418397862e-01 6.35976842802277175792369234841317e-03 3.43799094677205022488331032093356e-03
            6.63643196738745944074366889253724e-01 6.35976842802277175792369234841317e-03 3.29997034833231284167709418397862e-01 3.43799094677205022488331032093356e-03
            6.35976842802277175792369234841317e-03 3.29997034833231284167709418397862e-01 6.63643196738745944074366889253724e-01 3.43799094677205022488331032093356e-03
            6.35976842802277175792369234841317e-03 6.63643196738745944074366889253724e-01 3.29997034833231284167709418397862e-01 3.43799094677205022488331032093356e-03
            2.30829591569331249445440334966406e-01 7.63088635213633348364226094417972e-01 6.08177321703540219033357061562128e-03 2.86153444220796924005179029393275e-03
            2.30829591569331249445440334966406e-01 6.08177321703540219033357061562128e-03 7.63088635213633348364226094417972e-01 2.86153444220796924005179029393275e-03
            7.63088635213633348364226094417972e-01 2.30829591569331249445440334966406e-01 6.08177321703540219033357061562128e-03 2.86153444220796924005179029393275e-03
            7.63088635213633348364226094417972e-01 6.08177321703540219033357061562128e-03 2.30829591569331249445440334966406e-01 2.86153444220796924005179029393275e-03
            6.08177321703540219033357061562128e-03 2.30829591569331249445440334966406e-01 7.63088635213633348364226094417972e-01 2.86153444220796924005179029393275e-03
            6.08177321703540219033357061562128e-03 7.63088635213633348364226094417972e-01 2.30829591569331249445440334966406e-01 2.86153444220796924005179029393275e-03
            1.47944114085407502079405617223529e-01 8.46742931842361046257394718850264e-01 5.31295407223142390762404829729348e-03 2.03926220043675531998217387297245e-03
            1.47944114085407502079405617223529e-01 5.31295407223142390762404829729348e-03 8.46742931842361046257394718850264e-01 2.03926220043675531998217387297245e-03
            8.46742931842361046257394718850264e-01 1.47944114085407502079405617223529e-01 5.31295407223142390762404829729348e-03 2.03926220043675531998217387297245e-03
            8.46742931842361046257394718850264e-01 5.31295407223142390762404829729348e-03 1.47944114085407502079405617223529e-01 2.03926220043675531998217387297245e-03
            5.31295407223142390762404829729348e-03 1.47944114085407502079405617223529e-01 8.46742931842361046257394718850264e-01 2.03926220043675531998217387297245e-03
            5.31295407223142390762404829729348e-03 8.46742931842361046257394718850264e-01 1.47944114085407502079405617223529e-01 2.03926220043675531998217387297245e-03
            8.33105116593902267352333979033574e-02 9.09277521720756043421829417638946e-01 7.41196661985377147630060790106654e-03 2.12260415935469322692985549849709e-03
            8.33105116593902267352333979033574e-02 7.41196661985377147630060790106654e-03 9.09277521720756043421829417638946e-01 2.12260415935469322692985549849709e-03
            9.09277521720756043421829417638946e-01 8.33105116593902267352333979033574e-02 7.41196661985377147630060790106654e-03 2.12260415935469322692985549849709e-03
            9.09277521720756043421829417638946e-01 7.41196661985377147630060790106654e-03 8.33105116593902267352333979033574e-02 2.12260415935469322692985549849709e-03
            7.41196661985377147630060790106654e-03 8.33105116593902267352333979033574e-02 9.09277521720756043421829417638946e-01 2.12260415935469322692985549849709e-03
            7.41196661985377147630060790106654e-03 9.09277521720756043421829417638946e-01 8.33105116593902267352333979033574e-02 2.12260415935469322692985549849709e-03
            3.55026032675821653916692355323903e-02 9.58595570159293530743127575988183e-01 5.90182657312432468188490020111203e-03 1.19562615477588674896269882452771e-03
            3.55026032675821653916692355323903e-02 5.90182657312432468188490020111203e-03 9.58595570159293530743127575988183e-01 1.19562615477588674896269882452771e-03
            9.58595570159293530743127575988183e-01 3.55026032675821653916692355323903e-02 5.90182657312432468188490020111203e-03 1.19562615477588674896269882452771e-03
            9.58595570159293530743127575988183e-01 5.90182657312432468188490020111203e-03 3.55026032675821653916692355323903e-02 1.19562615477588674896269882452771e-03
            5.90182657312432468188490020111203e-03 3.55026032675821653916692355323903e-02 9.58595570159293530743127575988183e-01 1.19562615477588674896269882452771e-03
            5.90182657312432468188490020111203e-03 9.58595570159293530743127575988183e-01 3.55026032675821653916692355323903e-02 1.19562615477588674896269882452771e-03
            4.37992850963974222899821597820846e-01 5.28750100020195179517656924872426e-01 3.32570490158305975825214773067273e-02 6.85739162489405577882717324200712e-03
            4.37992850963974222899821597820846e-01 3.32570490158305975825214773067273e-02 5.28750100020195179517656924872426e-01 6.85739162489405577882717324200712e-03
            5.28750100020195179517656924872426e-01 4.37992850963974222899821597820846e-01 3.32570490158305975825214773067273e-02 6.85739162489405577882717324200712e-03
            5.28750100020195179517656924872426e-01 3.32570490158305975825214773067273e-02 4.37992850963974222899821597820846e-01 6.85739162489405577882717324200712e-03
            3.32570490158305975825214773067273e-02 4.37992850963974222899821597820846e-01 5.28750100020195179517656924872426e-01 6.85739162489405577882717324200712e-03
            3.32570490158305975825214773067273e-02 5.28750100020195179517656924872426e-01 4.37992850963974222899821597820846e-01 6.85739162489405577882717324200712e-03
            3.41845115301239788063725200117915e-01 6.25010662557993978971637716313126e-01 3.31442221407661774534858523111325e-02 7.34359958211473935751900654622659e-03
            3.41845115301239788063725200117915e-01 3.31442221407661774534858523111325e-02 6.25010662557993978971637716313126e-01 7.34359958211473935751900654622659e-03
            6.25010662557993978971637716313126e-01 3.41845115301239788063725200117915e-01 3.31442221407661774534858523111325e-02 7.34359958211473935751900654622659e-03
            6.25010662557993978971637716313126e-01 3.31442221407661774534858523111325e-02 3.41845115301239788063725200117915e-01 7.34359958211473935751900654622659e-03
            3.31442221407661774534858523111325e-02 3.41845115301239788063725200117915e-01 6.25010662557993978971637716313126e-01 7.34359958211473935751900654622659e-03
            3.31442221407661774534858523111325e-02 6.25010662557993978971637716313126e-01 3.41845115301239788063725200117915e-01 7.34359958211473935751900654622659e-03
            2.43219913165517859221509411327133e-01 7.24644701363325371978874045453267e-01 3.21353854711567965551921588485129e-02 6.92380910755335896716866628253229e-03
            2.43219913165517859221509411327133e-01 3.21353854711567965551921588485129e-02 7.24644701363325371978874045453267e-01 6.92380910755335896716866628253229e-03
            7.24644701363325371978874045453267e-01 2.43219913165517859221509411327133e-01 3.21353854711567965551921588485129e-02 6.92380910755335896716866628253229e-03
            7.24644701363325371978874045453267e-01 3.21353854711567965551921588485129e-02 2.43219913165517859221509411327133e-01 6.92380910755335896716866628253229e-03
            3.21353854711567965551921588485129e-02 2.43219913165517859221509411327133e-01 7.24644701363325371978874045453267e-01 6.92380910755335896716866628253229e-03
            3.21353854711567965551921588485129e-02 7.24644701363325371978874045453267e-01 2.43219913165517859221509411327133e-01 6.92380910755335896716866628253229e-03
            1.53693911780638098329987428769527e-01 8.16687808118266356238734715589089e-01 2.96182801010955731868534712702967e-02 5.68034562903817815543439095904432e-03
            1.53693911780638098329987428769527e-01 2.96182801010955731868534712702967e-02 8.16687808118266356238734715589089e-01 5.68034562903817815543439095904432e-03
            8.16687808118266356238734715589089e-01 1.53693911780638098329987428769527e-01 2.96182801010955731868534712702967e-02 5.68034562903817815543439095904432e-03
            8.16687808118266356238734715589089e-01 2.96182801010955731868534712702967e-02 1.53693911780638098329987428769527e-01 5.68034562903817815543439095904432e-03
            2.96182801010955731868534712702967e-02 1.53693911780638098329987428769527e-01 8.16687808118266356238734715589089e-01 5.68034562903817815543439095904432e-03
            2.96182801010955731868534712702967e-02 8.16687808118266356238734715589089e-01 1.53693911780638098329987428769527e-01 5.68034562903817815543439095904432e-03
            8.18282484051558456350505821319530e-02 8.81377376506472409722903194051469e-01 3.67943750883718001531974550744053e-02 4.82902212700271008666286221000519e-03
            8.18282484051558456350505821319530e-02 3.67943750883718001531974550744053e-02 8.81377376506472409722903194051469e-01 4.82902212700271008666286221000519e-03
            8.81377376506472409722903194051469e-01 8.18282484051558456350505821319530e-02 3.67943750883718001531974550744053e-02 4.82902212700271008666286221000519e-03
            8.81377376506472409722903194051469e-01 3.67943750883718001531974550744053e-02 8.18282484051558456350505821319530e-02 4.82902212700271008666286221000519e-03
            3.67943750883718001531974550744053e-02 8.18282484051558456350505821319530e-02 8.81377376506472409722903194051469e-01 4.82902212700271008666286221000519e-03
            3.67943750883718001531974550744053e-02 8.81377376506472409722903194051469e-01 8.18282484051558456350505821319530e-02 4.82902212700271008666286221000519e-03
            4.05435287023867385602926560750348e-01 5.14442885044733677624151368945604e-01 8.01218279313989922840733015618753e-02 1.21701611161561130264319174898446e-02
            4.05435287023867385602926560750348e-01 8.01218279313989922840733015618753e-02 5.14442885044733677624151368945604e-01 1.21701611161561130264319174898446e-02
            5.14442885044733677624151368945604e-01 4.05435287023867385602926560750348e-01 8.01218279313989922840733015618753e-02 1.21701611161561130264319174898446e-02
            5.14442885044733677624151368945604e-01 8.01218279313989922840733015618753e-02 4.05435287023867385602926560750348e-01 1.21701611161561130264319174898446e-02
            8.01218279313989922840733015618753e-02 4.05435287023867385602926560750348e-01 5.14442885044733677624151368945604e-01 1.21701611161561130264319174898446e-02
            8.01218279313989922840733015618753e-02 5.14442885044733677624151368945604e-01 4.05435287023867385602926560750348e-01 1.21701611161561130264319174898446e-02
            3.00475234451001216307020058593480e-01 6.20596936642523910876434456440620e-01 7.89278289064748728165454849659000e-02 1.09339847002783239343637333718107e-02
            3.00475234451001216307020058593480e-01 7.89278289064748728165454849659000e-02 6.20596936642523910876434456440620e-01 1.09339847002783239343637333718107e-02
            6.20596936642523910876434456440620e-01 3.00475234451001216307020058593480e-01 7.89278289064748728165454849659000e-02 1.09339847002783239343637333718107e-02
            6.20596936642523910876434456440620e-01 7.89278289064748728165454849659000e-02 3.00475234451001216307020058593480e-01 1.09339847002783239343637333718107e-02
            7.89278289064748728165454849659000e-02 3.00475234451001216307020058593480e-01 6.20596936642523910876434456440620e-01 1.09339847002783239343637333718107e-02
            7.89278289064748728165454849659000e-02 6.20596936642523910876434456440620e-01 3.00475234451001216307020058593480e-01 1.09339847002783239343637333718107e-02
            2.10298774983056657728042182498029e-01 7.12604684013138367504325287882239e-01 7.70965410038049192564812983619049e-02 8.74472678661106922470835911553877e-03
            2.10298774983056657728042182498029e-01 7.70965410038049192564812983619049e-02 7.12604684013138367504325287882239e-01 8.74472678661106922470835911553877e-03
            7.12604684013138367504325287882239e-01 2.10298774983056657728042182498029e-01 7.70965410038049192564812983619049e-02 8.74472678661106922470835911553877e-03
            7.12604684013138367504325287882239e-01 7.70965410038049192564812983619049e-02 2.10298774983056657728042182498029e-01 8.74472678661106922470835911553877e-03
            7.70965410038049192564812983619049e-02 2.10298774983056657728042182498029e-01 7.12604684013138367504325287882239e-01 8.74472678661106922470835911553877e-03
            7.70965410038049192564812983619049e-02 7.12604684013138367504325287882239e-01 2.10298774983056657728042182498029e-01 8.74472678661106922470835911553877e-03
            1.41929497236802870663296971542877e-01 7.83644226369100249662835722119780e-01 7.44262763940969351850185375951696e-02 6.80604517250698974606404334508625e-03
            1.41929497236802870663296971542877e-01 7.44262763940969351850185375951696e-02 7.83644226369100249662835722119780e-01 6.80604517250698974606404334508625e-03
            7.83644226369100249662835722119780e-01 1.41929497236802870663296971542877e-01 7.44262763940969351850185375951696e-02 6.80604517250698974606404334508625e-03
            7.83644226369100249662835722119780e-01 7.44262763940969351850185375951696e-02 1.41929497236802870663296971542877e-01 6.80604517250698974606404334508625e-03
            7.44262763940969351850185375951696e-02 1.41929497236802870663296971542877e-01 7.83644226369100249662835722119780e-01 6.80604517250698974606404334508625e-03
            7.44262763940969351850185375951696e-02 7.83644226369100249662835722119780e-01 1.41929497236802870663296971542877e-01 6.80604517250698974606404334508625e-03
            3.22825706055321792753431964229094e-01 5.35389696333471332145848009531619e-01 1.41784597611206875100720026239287e-01 1.41159091974328871482713410046017e-02
            3.22825706055321792753431964229094e-01 1.41784597611206875100720026239287e-01 5.35389696333471332145848009531619e-01 1.41159091974328871482713410046017e-02
            5.35389696333471332145848009531619e-01 3.22825706055321792753431964229094e-01 1.41784597611206875100720026239287e-01 1.41159091974328871482713410046017e-02
            5.35389696333471332145848009531619e-01 1.41784597611206875100720026239287e-01 3.22825706055321792753431964229094e-01 1.41159091974328871482713410046017e-02
            1.41784597611206875100720026239287e-01 3.22825706055321792753431964229094e-01 5.35389696333471332145848009531619e-01 1.41159091974328871482713410046017e-02
            1.41784597611206875100720026239287e-01 5.35389696333471332145848009531619e-01 3.22825706055321792753431964229094e-01 1.41159091974328871482713410046017e-02
            2.25038500867665025584329896446434e-01 6.33993341119415743101228599698516e-01 1.40968158012919286825592735112878e-01 1.31147856257194609513883065687878e-02
            2.25038500867665025584329896446434e-01 1.40968158012919286825592735112878e-01 6.33993341119415743101228599698516e-01 1.31147856257194609513883065687878e-02
            6.33993341119415743101228599698516e-01 2.25038500867665025584329896446434e-01 1.40968158012919286825592735112878e-01 1.31147856257194609513883065687878e-02
            6.33993341119415743101228599698516e-01 1.40968158012919286825592735112878e-01 2.25038500867665025584329896446434e-01 1.31147856257194609513883065687878e-02
            1.40968158012919286825592735112878e-01 2.25038500867665025584329896446434e-01 6.33993341119415743101228599698516e-01 1.31147856257194609513883065687878e-02
            1.40968158012919286825592735112878e-01 6.33993341119415743101228599698516e-01 2.25038500867665025584329896446434e-01 1.31147856257194609513883065687878e-02
            3.06228014522925306994949323780020e-01 4.80239008804562494958645402221009e-01 2.13532976672512253557556505256798e-01 1.29403394597649937930450292355999e-02
            3.06228014522925306994949323780020e-01 2.13532976672512253557556505256798e-01 4.80239008804562494958645402221009e-01 1.29403394597649937930450292355999e-02
            4.80239008804562494958645402221009e-01 3.06228014522925306994949323780020e-01 2.13532976672512253557556505256798e-01 1.29403394597649937930450292355999e-02
            4.80239008804562494958645402221009e-01 2.13532976672512253557556505256798e-01 3.06228014522925306994949323780020e-01 1.29403394597649937930450292355999e-02
            2.13532976672512253557556505256798e-01 3.06228014522925306994949323780020e-01 4.80239008804562494958645402221009e-01 1.29403394597649937930450292355999e-02
            2.13532976672512253557556505256798e-01 4.80239008804562494958645402221009e-01 3.06228014522925306994949323780020e-01 1.29403394597649937930450292355999e-02
            ];
        
    case  27
        
        % ALG. DEG.:   27
        % PTS CARD.:  145
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.0e-15
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 1.32645422676788324843621946769190e-03
            4.64270848054669227344959381298395e-01 4.64270848054669227344959381298395e-01 7.14583038906615453100812374032103e-02 1.01760677725488100919815082079367e-02
            4.64270848054669227344959381298395e-01 7.14583038906615453100812374032103e-02 4.64270848054669227344959381298395e-01 1.01760677725488100919815082079367e-02
            7.14583038906615453100812374032103e-02 4.64270848054669227344959381298395e-01 4.64270848054669227344959381298395e-01 1.01760677725488100919815082079367e-02
            4.32298314340281597534243474001414e-01 4.32298314340281597534243474001414e-01 1.35403371319436804931513051997172e-01 1.35536351895579314924544789278116e-02
            4.32298314340281597534243474001414e-01 1.35403371319436804931513051997172e-01 4.32298314340281597534243474001414e-01 1.35536351895579314924544789278116e-02
            1.35403371319436804931513051997172e-01 4.32298314340281597534243474001414e-01 4.32298314340281597534243474001414e-01 1.35536351895579314924544789278116e-02
            3.90753814771931773375257535008132e-01 3.90753814771931773375257535008132e-01 2.18492370456136453249484929983737e-01 1.43557523235289946628157053964969e-02
            3.90753814771931773375257535008132e-01 2.18492370456136453249484929983737e-01 3.90753814771931773375257535008132e-01 1.43557523235289946628157053964969e-02
            2.18492370456136453249484929983737e-01 3.90753814771931773375257535008132e-01 3.90753814771931773375257535008132e-01 1.43557523235289946628157053964969e-02
            2.99849030838122532216516447078902e-01 2.99849030838122532216516447078902e-01 4.00301938323754935566967105842195e-01 1.76215402547928515208930377866636e-02
            2.99849030838122532216516447078902e-01 4.00301938323754935566967105842195e-01 2.99849030838122532216516447078902e-01 1.76215402547928515208930377866636e-02
            4.00301938323754935566967105842195e-01 2.99849030838122532216516447078902e-01 2.99849030838122532216516447078902e-01 1.76215402547928515208930377866636e-02
            2.42386093366695004602817675731785e-01 2.42386093366695004602817675731785e-01 5.15227813266609935283213417278603e-01 1.28645661240754105580919031126541e-02
            2.42386093366695004602817675731785e-01 5.15227813266609935283213417278603e-01 2.42386093366695004602817675731785e-01 1.28645661240754105580919031126541e-02
            5.15227813266609935283213417278603e-01 2.42386093366695004602817675731785e-01 2.42386093366695004602817675731785e-01 1.28645661240754105580919031126541e-02
            1.95751633844707950693475595471682e-01 1.95751633844707950693475595471682e-01 6.08496732310584098613048809056636e-01 1.02112078872144251134868042640846e-02
            1.95751633844707950693475595471682e-01 6.08496732310584098613048809056636e-01 1.95751633844707950693475595471682e-01 1.02112078872144251134868042640846e-02
            6.08496732310584098613048809056636e-01 1.95751633844707950693475595471682e-01 1.95751633844707950693475595471682e-01 1.02112078872144251134868042640846e-02
            1.10964220718018535816185021758429e-01 1.10964220718018535816185021758429e-01 7.78071558563962928367629956483142e-01 8.05484559136441721771682011876692e-03
            1.10964220718018535816185021758429e-01 7.78071558563962928367629956483142e-01 1.10964220718018535816185021758429e-01 8.05484559136441721771682011876692e-03
            7.78071558563962928367629956483142e-01 1.10964220718018535816185021758429e-01 1.10964220718018535816185021758429e-01 8.05484559136441721771682011876692e-03
            6.78494378207233628241823453208781e-02 6.78494378207233628241823453208781e-02 8.64301124358553218840484078100417e-01 4.85083247712445216059951036413622e-03
            6.78494378207233628241823453208781e-02 8.64301124358553218840484078100417e-01 6.78494378207233628241823453208781e-02 4.85083247712445216059951036413622e-03
            8.64301124358553218840484078100417e-01 6.78494378207233628241823453208781e-02 6.78494378207233628241823453208781e-02 4.85083247712445216059951036413622e-03
            3.24881443144369105868740632558911e-02 3.24881443144369105868740632558911e-02 9.35023711371126164948464065673761e-01 2.76034923845063775646457848722548e-03
            3.24881443144369105868740632558911e-02 9.35023711371126164948464065673761e-01 3.24881443144369105868740632558911e-02 2.76034923845063775646457848722548e-03
            9.35023711371126164948464065673761e-01 3.24881443144369105868740632558911e-02 3.24881443144369105868740632558911e-02 2.76034923845063775646457848722548e-03
            7.02227048619495294867753898415685e-03 7.02227048619495294867753898415685e-03 9.85955459027610059408175402495544e-01 6.46520224024233879783762901638511e-04
            7.02227048619495294867753898415685e-03 9.85955459027610059408175402495544e-01 7.02227048619495294867753898415685e-03 6.46520224024233879783762901638511e-04
            9.85955459027610059408175402495544e-01 7.02227048619495294867753898415685e-03 7.02227048619495294867753898415685e-03 6.46520224024233879783762901638511e-04
            4.44334487696719404592471391879371e-01 5.50169162481872864312038018397288e-01 5.49634982140767558433935846551321e-03 2.98280618704304311891717382820843e-03
            4.44334487696719404592471391879371e-01 5.49634982140767558433935846551321e-03 5.50169162481872864312038018397288e-01 2.98280618704304311891717382820843e-03
            5.50169162481872864312038018397288e-01 4.44334487696719404592471391879371e-01 5.49634982140767558433935846551321e-03 2.98280618704304311891717382820843e-03
            5.50169162481872864312038018397288e-01 5.49634982140767558433935846551321e-03 4.44334487696719404592471391879371e-01 2.98280618704304311891717382820843e-03
            5.49634982140767558433935846551321e-03 4.44334487696719404592471391879371e-01 5.50169162481872864312038018397288e-01 2.98280618704304311891717382820843e-03
            5.49634982140767558433935846551321e-03 5.50169162481872864312038018397288e-01 4.44334487696719404592471391879371e-01 2.98280618704304311891717382820843e-03
            2.42719887748324536502764203760307e-01 7.51198680548561092962245311355218e-01 6.08143170311437053499048488447443e-03 2.86890029157565919909478324711927e-03
            2.42719887748324536502764203760307e-01 6.08143170311437053499048488447443e-03 7.51198680548561092962245311355218e-01 2.86890029157565919909478324711927e-03
            7.51198680548561092962245311355218e-01 2.42719887748324536502764203760307e-01 6.08143170311437053499048488447443e-03 2.86890029157565919909478324711927e-03
            7.51198680548561092962245311355218e-01 6.08143170311437053499048488447443e-03 2.42719887748324536502764203760307e-01 2.86890029157565919909478324711927e-03
            6.08143170311437053499048488447443e-03 2.42719887748324536502764203760307e-01 7.51198680548561092962245311355218e-01 2.86890029157565919909478324711927e-03
            6.08143170311437053499048488447443e-03 7.51198680548561092962245311355218e-01 2.42719887748324536502764203760307e-01 2.86890029157565919909478324711927e-03
            1.58188924752096032877446418751788e-01 8.38350695312842342232784176303539e-01 3.46037993506165264534502057358623e-03 1.46771786779737465136741469962089e-03
            1.58188924752096032877446418751788e-01 3.46037993506165264534502057358623e-03 8.38350695312842342232784176303539e-01 1.46771786779737465136741469962089e-03
            8.38350695312842342232784176303539e-01 1.58188924752096032877446418751788e-01 3.46037993506165264534502057358623e-03 1.46771786779737465136741469962089e-03
            8.38350695312842342232784176303539e-01 3.46037993506165264534502057358623e-03 1.58188924752096032877446418751788e-01 1.46771786779737465136741469962089e-03
            3.46037993506165264534502057358623e-03 1.58188924752096032877446418751788e-01 8.38350695312842342232784176303539e-01 1.46771786779737465136741469962089e-03
            3.46037993506165264534502057358623e-03 8.38350695312842342232784176303539e-01 1.58188924752096032877446418751788e-01 1.46771786779737465136741469962089e-03
            8.84556064909017975006477740862465e-02 9.06294109675106485113360577088315e-01 5.25028383399173126377945663989522e-03 1.63545643437809748617361993439090e-03
            8.84556064909017975006477740862465e-02 5.25028383399173126377945663989522e-03 9.06294109675106485113360577088315e-01 1.63545643437809748617361993439090e-03
            9.06294109675106485113360577088315e-01 8.84556064909017975006477740862465e-02 5.25028383399173126377945663989522e-03 1.63545643437809748617361993439090e-03
            9.06294109675106485113360577088315e-01 5.25028383399173126377945663989522e-03 8.84556064909017975006477740862465e-02 1.63545643437809748617361993439090e-03
            5.25028383399173126377945663989522e-03 8.84556064909017975006477740862465e-02 9.06294109675106485113360577088315e-01 1.63545643437809748617361993439090e-03
            5.25028383399173126377945663989522e-03 9.06294109675106485113360577088315e-01 8.84556064909017975006477740862465e-02 1.63545643437809748617361993439090e-03
            3.39993168915069976510068272546050e-01 6.53721870992564402236268961132737e-01 6.28496009236567676481399757904001e-03 3.26270663499477874686105494106414e-03
            3.39993168915069976510068272546050e-01 6.28496009236567676481399757904001e-03 6.53721870992564402236268961132737e-01 3.26270663499477874686105494106414e-03
            6.53721870992564402236268961132737e-01 3.39993168915069976510068272546050e-01 6.28496009236567676481399757904001e-03 3.26270663499477874686105494106414e-03
            6.53721870992564402236268961132737e-01 6.28496009236567676481399757904001e-03 3.39993168915069976510068272546050e-01 3.26270663499477874686105494106414e-03
            6.28496009236567676481399757904001e-03 3.39993168915069976510068272546050e-01 6.53721870992564402236268961132737e-01 3.26270663499477874686105494106414e-03
            6.28496009236567676481399757904001e-03 6.53721870992564402236268961132737e-01 3.39993168915069976510068272546050e-01 3.26270663499477874686105494106414e-03
            3.67115484722965942920858140041673e-02 9.57246043701164284200899601273704e-01 6.04240782653908681254506518598646e-03 1.27830372794885890885829837770871e-03
            3.67115484722965942920858140041673e-02 6.04240782653908681254506518598646e-03 9.57246043701164284200899601273704e-01 1.27830372794885890885829837770871e-03
            9.57246043701164284200899601273704e-01 3.67115484722965942920858140041673e-02 6.04240782653908681254506518598646e-03 1.27830372794885890885829837770871e-03
            9.57246043701164284200899601273704e-01 6.04240782653908681254506518598646e-03 3.67115484722965942920858140041673e-02 1.27830372794885890885829837770871e-03
            6.04240782653908681254506518598646e-03 3.67115484722965942920858140041673e-02 9.57246043701164284200899601273704e-01 1.27830372794885890885829837770871e-03
            6.04240782653908681254506518598646e-03 9.57246043701164284200899601273704e-01 3.67115484722965942920858140041673e-02 1.27830372794885890885829837770871e-03
            8.12402961036135656813783612051338e-02 8.91779193842819672966015787096694e-01 2.69805100535667197192424282548018e-02 3.55054931897130863355238616918541e-03
            8.12402961036135656813783612051338e-02 2.69805100535667197192424282548018e-02 8.91779193842819672966015787096694e-01 3.55054931897130863355238616918541e-03
            8.91779193842819672966015787096694e-01 8.12402961036135656813783612051338e-02 2.69805100535667197192424282548018e-02 3.55054931897130863355238616918541e-03
            8.91779193842819672966015787096694e-01 2.69805100535667197192424282548018e-02 8.12402961036135656813783612051338e-02 3.55054931897130863355238616918541e-03
            2.69805100535667197192424282548018e-02 8.12402961036135656813783612051338e-02 8.91779193842819672966015787096694e-01 3.55054931897130863355238616918541e-03
            2.69805100535667197192424282548018e-02 8.91779193842819672966015787096694e-01 8.12402961036135656813783612051338e-02 3.55054931897130863355238616918541e-03
            4.31949991404557809282493963110028e-01 5.38923331029669272496107623737771e-01 2.91266775657729182213984131522011e-02 7.05687847901615827178600781621753e-03
            4.31949991404557809282493963110028e-01 2.91266775657729182213984131522011e-02 5.38923331029669272496107623737771e-01 7.05687847901615827178600781621753e-03
            5.38923331029669272496107623737771e-01 4.31949991404557809282493963110028e-01 2.91266775657729182213984131522011e-02 7.05687847901615827178600781621753e-03
            5.38923331029669272496107623737771e-01 2.91266775657729182213984131522011e-02 4.31949991404557809282493963110028e-01 7.05687847901615827178600781621753e-03
            2.91266775657729182213984131522011e-02 4.31949991404557809282493963110028e-01 5.38923331029669272496107623737771e-01 7.05687847901615827178600781621753e-03
            2.91266775657729182213984131522011e-02 5.38923331029669272496107623737771e-01 4.31949991404557809282493963110028e-01 7.05687847901615827178600781621753e-03
            3.26996931617198338670959856244735e-01 6.40470902013947407382943310949486e-01 3.25321663688542539460968328057788e-02 7.10874754932739119639428793107072e-03
            3.26996931617198338670959856244735e-01 3.25321663688542539460968328057788e-02 6.40470902013947407382943310949486e-01 7.10874754932739119639428793107072e-03
            6.40470902013947407382943310949486e-01 3.26996931617198338670959856244735e-01 3.25321663688542539460968328057788e-02 7.10874754932739119639428793107072e-03
            6.40470902013947407382943310949486e-01 3.25321663688542539460968328057788e-02 3.26996931617198338670959856244735e-01 7.10874754932739119639428793107072e-03
            3.25321663688542539460968328057788e-02 3.26996931617198338670959856244735e-01 6.40470902013947407382943310949486e-01 7.10874754932739119639428793107072e-03
            3.25321663688542539460968328057788e-02 6.40470902013947407382943310949486e-01 3.26996931617198338670959856244735e-01 7.10874754932739119639428793107072e-03
            2.34081170727359955119339929296984e-01 7.34315971485084095782269741903292e-01 3.16028577875558935872390975418966e-02 6.19331568447357115603235655498793e-03
            2.34081170727359955119339929296984e-01 3.16028577875558935872390975418966e-02 7.34315971485084095782269741903292e-01 6.19331568447357115603235655498793e-03
            7.34315971485084095782269741903292e-01 2.34081170727359955119339929296984e-01 3.16028577875558935872390975418966e-02 6.19331568447357115603235655498793e-03
            7.34315971485084095782269741903292e-01 3.16028577875558935872390975418966e-02 2.34081170727359955119339929296984e-01 6.19331568447357115603235655498793e-03
            3.16028577875558935872390975418966e-02 2.34081170727359955119339929296984e-01 7.34315971485084095782269741903292e-01 6.19331568447357115603235655498793e-03
            3.16028577875558935872390975418966e-02 7.34315971485084095782269741903292e-01 2.34081170727359955119339929296984e-01 6.19331568447357115603235655498793e-03
            1.53473632723084674189806264621438e-01 8.24922676183995506882240533741424e-01 2.16036910929198189279532016371377e-02 4.30573263634848740766258856638160e-03
            1.53473632723084674189806264621438e-01 2.16036910929198189279532016371377e-02 8.24922676183995506882240533741424e-01 4.30573263634848740766258856638160e-03
            8.24922676183995506882240533741424e-01 1.53473632723084674189806264621438e-01 2.16036910929198189279532016371377e-02 4.30573263634848740766258856638160e-03
            8.24922676183995506882240533741424e-01 2.16036910929198189279532016371377e-02 1.53473632723084674189806264621438e-01 4.30573263634848740766258856638160e-03
            2.16036910929198189279532016371377e-02 1.53473632723084674189806264621438e-01 8.24922676183995506882240533741424e-01 4.30573263634848740766258856638160e-03
            2.16036910929198189279532016371377e-02 8.24922676183995506882240533741424e-01 1.53473632723084674189806264621438e-01 4.30573263634848740766258856638160e-03
            1.26999415890300532128165400536091e-01 8.16884834249502755909588813665323e-01 5.61157498601967397178214014274999e-02 5.38152713709093982741293871185917e-03
            1.26999415890300532128165400536091e-01 5.61157498601967397178214014274999e-02 8.16884834249502755909588813665323e-01 5.38152713709093982741293871185917e-03
            8.16884834249502755909588813665323e-01 1.26999415890300532128165400536091e-01 5.61157498601967397178214014274999e-02 5.38152713709093982741293871185917e-03
            8.16884834249502755909588813665323e-01 5.61157498601967397178214014274999e-02 1.26999415890300532128165400536091e-01 5.38152713709093982741293871185917e-03
            5.61157498601967397178214014274999e-02 1.26999415890300532128165400536091e-01 8.16884834249502755909588813665323e-01 5.38152713709093982741293871185917e-03
            5.61157498601967397178214014274999e-02 8.16884834249502755909588813665323e-01 1.26999415890300532128165400536091e-01 5.38152713709093982741293871185917e-03
            3.68066602265455233844448912350344e-01 5.59340535857428933219637201545993e-01 7.25928618771157774247626548458356e-02 9.46147607081304804765853333492487e-03
            3.68066602265455233844448912350344e-01 7.25928618771157774247626548458356e-02 5.59340535857428933219637201545993e-01 9.46147607081304804765853333492487e-03
            5.59340535857428933219637201545993e-01 3.68066602265455233844448912350344e-01 7.25928618771157774247626548458356e-02 9.46147607081304804765853333492487e-03
            5.59340535857428933219637201545993e-01 7.25928618771157774247626548458356e-02 3.68066602265455233844448912350344e-01 9.46147607081304804765853333492487e-03
            7.25928618771157774247626548458356e-02 3.68066602265455233844448912350344e-01 5.59340535857428933219637201545993e-01 9.46147607081304804765853333492487e-03
            7.25928618771157774247626548458356e-02 5.59340535857428933219637201545993e-01 3.68066602265455233844448912350344e-01 9.46147607081304804765853333492487e-03
            2.69796071148512450754708424938144e-01 6.50885906365806787832184454600792e-01 7.93180224856807614131071204610635e-02 1.03877644118448696913814544018351e-02
            2.69796071148512450754708424938144e-01 7.93180224856807614131071204610635e-02 6.50885906365806787832184454600792e-01 1.03877644118448696913814544018351e-02
            6.50885906365806787832184454600792e-01 2.69796071148512450754708424938144e-01 7.93180224856807614131071204610635e-02 1.03877644118448696913814544018351e-02
            6.50885906365806787832184454600792e-01 7.93180224856807614131071204610635e-02 2.69796071148512450754708424938144e-01 1.03877644118448696913814544018351e-02
            7.93180224856807614131071204610635e-02 2.69796071148512450754708424938144e-01 6.50885906365806787832184454600792e-01 1.03877644118448696913814544018351e-02
            7.93180224856807614131071204610635e-02 6.50885906365806787832184454600792e-01 2.69796071148512450754708424938144e-01 1.03877644118448696913814544018351e-02
            1.86994710857735058073814116141875e-01 7.44996011981384320677079813322052e-01 6.80092771608806767602573017938994e-02 6.93741437082199402519577802195272e-03
            1.86994710857735058073814116141875e-01 6.80092771608806767602573017938994e-02 7.44996011981384320677079813322052e-01 6.93741437082199402519577802195272e-03
            7.44996011981384320677079813322052e-01 1.86994710857735058073814116141875e-01 6.80092771608806767602573017938994e-02 6.93741437082199402519577802195272e-03
            7.44996011981384320677079813322052e-01 6.80092771608806767602573017938994e-02 1.86994710857735058073814116141875e-01 6.93741437082199402519577802195272e-03
            6.80092771608806767602573017938994e-02 1.86994710857735058073814116141875e-01 7.44996011981384320677079813322052e-01 6.93741437082199402519577802195272e-03
            6.80092771608806767602573017938994e-02 7.44996011981384320677079813322052e-01 1.86994710857735058073814116141875e-01 6.93741437082199402519577802195272e-03
            3.45950442872727914611630239960505e-01 5.27094080400327680280270215007477e-01 1.26955476726944405108099545032019e-01 1.05678608749397197341668430681239e-02
            3.45950442872727914611630239960505e-01 1.26955476726944405108099545032019e-01 5.27094080400327680280270215007477e-01 1.05678608749397197341668430681239e-02
            5.27094080400327680280270215007477e-01 3.45950442872727914611630239960505e-01 1.26955476726944405108099545032019e-01 1.05678608749397197341668430681239e-02
            5.27094080400327680280270215007477e-01 1.26955476726944405108099545032019e-01 3.45950442872727914611630239960505e-01 1.05678608749397197341668430681239e-02
            1.26955476726944405108099545032019e-01 3.45950442872727914611630239960505e-01 5.27094080400327680280270215007477e-01 1.05678608749397197341668430681239e-02
            1.26955476726944405108099545032019e-01 5.27094080400327680280270215007477e-01 3.45950442872727914611630239960505e-01 1.05678608749397197341668430681239e-02
            1.82456322380054797305248825978197e-01 6.93857687409184187288246903335676e-01 1.23685990210760987650928655057214e-01 1.00159152041039631320096248146001e-02
            1.82456322380054797305248825978197e-01 1.23685990210760987650928655057214e-01 6.93857687409184187288246903335676e-01 1.00159152041039631320096248146001e-02
            6.93857687409184187288246903335676e-01 1.82456322380054797305248825978197e-01 1.23685990210760987650928655057214e-01 1.00159152041039631320096248146001e-02
            6.93857687409184187288246903335676e-01 1.23685990210760987650928655057214e-01 1.82456322380054797305248825978197e-01 1.00159152041039631320096248146001e-02
            1.23685990210760987650928655057214e-01 1.82456322380054797305248825978197e-01 6.93857687409184187288246903335676e-01 1.00159152041039631320096248146001e-02
            1.23685990210760987650928655057214e-01 6.93857687409184187288246903335676e-01 1.82456322380054797305248825978197e-01 1.00159152041039631320096248146001e-02
            2.60183849659498422912662363160052e-01 5.94783798861356349796380982297705e-01 1.45032351479145282802107885800069e-01 1.14577693947056670487061325047762e-02
            2.60183849659498422912662363160052e-01 1.45032351479145282802107885800069e-01 5.94783798861356349796380982297705e-01 1.14577693947056670487061325047762e-02
            5.94783798861356349796380982297705e-01 2.60183849659498422912662363160052e-01 1.45032351479145282802107885800069e-01 1.14577693947056670487061325047762e-02
            5.94783798861356349796380982297705e-01 1.45032351479145282802107885800069e-01 2.60183849659498422912662363160052e-01 1.14577693947056670487061325047762e-02
            1.45032351479145282802107885800069e-01 2.60183849659498422912662363160052e-01 5.94783798861356349796380982297705e-01 1.14577693947056670487061325047762e-02
            1.45032351479145282802107885800069e-01 5.94783798861356349796380982297705e-01 2.60183849659498422912662363160052e-01 1.14577693947056670487061325047762e-02
            3.19848673583135789666442860834650e-01 4.82373460310693780161983568177675e-01 1.97777866106170374660422339729848e-01 1.29770901446694172237661391022812e-02
            3.19848673583135789666442860834650e-01 1.97777866106170374660422339729848e-01 4.82373460310693780161983568177675e-01 1.29770901446694172237661391022812e-02
            4.82373460310693780161983568177675e-01 3.19848673583135789666442860834650e-01 1.97777866106170374660422339729848e-01 1.29770901446694172237661391022812e-02
            4.82373460310693780161983568177675e-01 1.97777866106170374660422339729848e-01 3.19848673583135789666442860834650e-01 1.29770901446694172237661391022812e-02
            1.97777866106170374660422339729848e-01 3.19848673583135789666442860834650e-01 4.82373460310693780161983568177675e-01 1.29770901446694172237661391022812e-02
            1.97777866106170374660422339729848e-01 4.82373460310693780161983568177675e-01 3.19848673583135789666442860834650e-01 1.29770901446694172237661391022812e-02
            ];
        
    case  28
        
        % ALG. DEG.:   28
        % PTS CARD.:  154
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.1e-15
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.72851181690746560914995200164412e-03
            4.98353413126779720698777964571491e-01 4.98353413126779720698777964571491e-01 3.29317374644055860244407085701823e-03 1.93654415522498965854247821027911e-03
            4.98353413126779720698777964571491e-01 3.29317374644055860244407085701823e-03 4.98353413126779720698777964571491e-01 1.93654415522498965854247821027911e-03
            3.29317374644055860244407085701823e-03 4.98353413126779720698777964571491e-01 4.98353413126779720698777964571491e-01 1.93654415522498965854247821027911e-03
            4.88562754090508810644877257800545e-01 4.88562754090508810644877257800545e-01 2.28744918189823787102454843989108e-02 6.05141039266347911113053115172988e-03
            4.88562754090508810644877257800545e-01 2.28744918189823787102454843989108e-02 4.88562754090508810644877257800545e-01 6.05141039266347911113053115172988e-03
            2.28744918189823787102454843989108e-02 4.88562754090508810644877257800545e-01 4.88562754090508810644877257800545e-01 6.05141039266347911113053115172988e-03
            4.64781478287247362679579509858740e-01 4.64781478287247362679579509858740e-01 7.04370434255052746408409802825190e-02 1.14141135507166320561811545530873e-02
            4.64781478287247362679579509858740e-01 7.04370434255052746408409802825190e-02 4.64781478287247362679579509858740e-01 1.14141135507166320561811545530873e-02
            7.04370434255052746408409802825190e-02 4.64781478287247362679579509858740e-01 4.64781478287247362679579509858740e-01 1.14141135507166320561811545530873e-02
            3.89766273321017708841651483453461e-01 3.89766273321017708841651483453461e-01 2.20467453357964582316697033093078e-01 1.60800737631301184082932564933799e-02
            3.89766273321017708841651483453461e-01 2.20467453357964582316697033093078e-01 3.89766273321017708841651483453461e-01 1.60800737631301184082932564933799e-02
            2.20467453357964582316697033093078e-01 3.89766273321017708841651483453461e-01 3.89766273321017708841651483453461e-01 1.60800737631301184082932564933799e-02
            3.01546751182982020189626837236574e-01 3.01546751182982020189626837236574e-01 3.96906497634035959620746325526852e-01 1.40353291043240624436094776683603e-02
            3.01546751182982020189626837236574e-01 3.96906497634035959620746325526852e-01 3.01546751182982020189626837236574e-01 1.40353291043240624436094776683603e-02
            3.96906497634035959620746325526852e-01 3.01546751182982020189626837236574e-01 3.01546751182982020189626837236574e-01 1.40353291043240624436094776683603e-02
            2.61574074059258976365782700668206e-01 2.61574074059258976365782700668206e-01 4.76851851881482047268434598663589e-01 9.78327794286425536651741197147203e-03
            2.61574074059258976365782700668206e-01 4.76851851881482047268434598663589e-01 2.61574074059258976365782700668206e-01 9.78327794286425536651741197147203e-03
            4.76851851881482047268434598663589e-01 2.61574074059258976365782700668206e-01 2.61574074059258976365782700668206e-01 9.78327794286425536651741197147203e-03
            2.14107886345909520509067647253687e-01 2.14107886345909520509067647253687e-01 5.71784227308181014493015936750453e-01 1.26783016982055917604155226285911e-02
            2.14107886345909520509067647253687e-01 5.71784227308181014493015936750453e-01 2.14107886345909520509067647253687e-01 1.26783016982055917604155226285911e-02
            5.71784227308181014493015936750453e-01 2.14107886345909520509067647253687e-01 2.14107886345909520509067647253687e-01 1.26783016982055917604155226285911e-02
            1.70138191271475530674806009301392e-01 1.70138191271475530674806009301392e-01 6.59723617457048883139236750139389e-01 1.01141158594697383454885653009114e-02
            1.70138191271475530674806009301392e-01 6.59723617457048883139236750139389e-01 1.70138191271475530674806009301392e-01 1.01141158594697383454885653009114e-02
            6.59723617457048883139236750139389e-01 1.70138191271475530674806009301392e-01 1.70138191271475530674806009301392e-01 1.01141158594697383454885653009114e-02
            6.43707835026944286127559280430432e-02 6.43707835026944286127559280430432e-02 8.71258432994611142774488143913914e-01 3.76318422499544950193817172134914e-03
            6.43707835026944286127559280430432e-02 8.71258432994611142774488143913914e-01 6.43707835026944286127559280430432e-02 3.76318422499544950193817172134914e-03
            8.71258432994611142774488143913914e-01 6.43707835026944286127559280430432e-02 6.43707835026944286127559280430432e-02 3.76318422499544950193817172134914e-03
            3.05600321040132086314322634734708e-02 3.05600321040132086314322634734708e-02 9.38879935791973596614923280867515e-01 2.43119330615521956132085712454227e-03
            3.05600321040132086314322634734708e-02 9.38879935791973596614923280867515e-01 3.05600321040132086314322634734708e-02 2.43119330615521956132085712454227e-03
            9.38879935791973596614923280867515e-01 3.05600321040132086314322634734708e-02 3.05600321040132086314322634734708e-02 2.43119330615521956132085712454227e-03
            5.70940793308473588940632481580906e-03 5.70940793308473588940632481580906e-03 9.88581184133830559446209917950910e-01 4.25889945316463983723126451863550e-04
            5.70940793308473588940632481580906e-03 9.88581184133830559446209917950910e-01 5.70940793308473588940632481580906e-03 4.25889945316463983723126451863550e-04
            9.88581184133830559446209917950910e-01 5.70940793308473588940632481580906e-03 5.70940793308473588940632481580906e-03 4.25889945316463983723126451863550e-04
            2.95361534971414929628963363938965e-01 7.00853506852401775262251248932444e-01 3.78495817618329510878538712859154e-03 2.06901772064515404592199843136768e-03
            2.95361534971414929628963363938965e-01 3.78495817618329510878538712859154e-03 7.00853506852401775262251248932444e-01 2.06901772064515404592199843136768e-03
            7.00853506852401775262251248932444e-01 2.95361534971414929628963363938965e-01 3.78495817618329510878538712859154e-03 2.06901772064515404592199843136768e-03
            7.00853506852401775262251248932444e-01 3.78495817618329510878538712859154e-03 2.95361534971414929628963363938965e-01 2.06901772064515404592199843136768e-03
            3.78495817618329510878538712859154e-03 2.95361534971414929628963363938965e-01 7.00853506852401775262251248932444e-01 2.06901772064515404592199843136768e-03
            3.78495817618329510878538712859154e-03 7.00853506852401775262251248932444e-01 2.95361534971414929628963363938965e-01 2.06901772064515404592199843136768e-03
            2.02876702155597610222059756779345e-01 7.91935619745775754019234682345996e-01 5.18767809862663575870556087465957e-03 2.15432386268054297528973251019124e-03
            2.02876702155597610222059756779345e-01 5.18767809862663575870556087465957e-03 7.91935619745775754019234682345996e-01 2.15432386268054297528973251019124e-03
            7.91935619745775754019234682345996e-01 2.02876702155597610222059756779345e-01 5.18767809862663575870556087465957e-03 2.15432386268054297528973251019124e-03
            7.91935619745775754019234682345996e-01 5.18767809862663575870556087465957e-03 2.02876702155597610222059756779345e-01 2.15432386268054297528973251019124e-03
            5.18767809862663575870556087465957e-03 2.02876702155597610222059756779345e-01 7.91935619745775754019234682345996e-01 2.15432386268054297528973251019124e-03
            5.18767809862663575870556087465957e-03 7.91935619745775754019234682345996e-01 2.02876702155597610222059756779345e-01 2.15432386268054297528973251019124e-03
            1.28179495236313306660846933482389e-01 8.67015046983888204579216107958928e-01 4.80545777979846100436134292976931e-03 1.66065394763716882124693352551503e-03
            1.28179495236313306660846933482389e-01 4.80545777979846100436134292976931e-03 8.67015046983888204579216107958928e-01 1.66065394763716882124693352551503e-03
            8.67015046983888204579216107958928e-01 1.28179495236313306660846933482389e-01 4.80545777979846100436134292976931e-03 1.66065394763716882124693352551503e-03
            8.67015046983888204579216107958928e-01 4.80545777979846100436134292976931e-03 1.28179495236313306660846933482389e-01 1.66065394763716882124693352551503e-03
            4.80545777979846100436134292976931e-03 1.28179495236313306660846933482389e-01 8.67015046983888204579216107958928e-01 1.66065394763716882124693352551503e-03
            4.80545777979846100436134292976931e-03 8.67015046983888204579216107958928e-01 1.28179495236313306660846933482389e-01 1.66065394763716882124693352551503e-03
            7.05254410920549829056014345951553e-02 9.23919864914773625308441751258215e-01 5.55469399317143341932023759000003e-03 1.38084032913644979764788978116030e-03
            7.05254410920549829056014345951553e-02 5.55469399317143341932023759000003e-03 9.23919864914773625308441751258215e-01 1.38084032913644979764788978116030e-03
            9.23919864914773625308441751258215e-01 7.05254410920549829056014345951553e-02 5.55469399317143341932023759000003e-03 1.38084032913644979764788978116030e-03
            9.23919864914773625308441751258215e-01 5.55469399317143341932023759000003e-03 7.05254410920549829056014345951553e-02 1.38084032913644979764788978116030e-03
            5.55469399317143341932023759000003e-03 7.05254410920549829056014345951553e-02 9.23919864914773625308441751258215e-01 1.38084032913644979764788978116030e-03
            5.55469399317143341932023759000003e-03 9.23919864914773625308441751258215e-01 7.05254410920549829056014345951553e-02 1.38084032913644979764788978116030e-03
            3.95326317408967009559006555718952e-01 5.97713061106773357877841590379830e-01 6.96062148425968807430308515904471e-03 3.60171025661803036535535404993880e-03
            3.95326317408967009559006555718952e-01 6.96062148425968807430308515904471e-03 5.97713061106773357877841590379830e-01 3.60171025661803036535535404993880e-03
            5.97713061106773357877841590379830e-01 3.95326317408967009559006555718952e-01 6.96062148425968807430308515904471e-03 3.60171025661803036535535404993880e-03
            5.97713061106773357877841590379830e-01 6.96062148425968807430308515904471e-03 3.95326317408967009559006555718952e-01 3.60171025661803036535535404993880e-03
            6.96062148425968807430308515904471e-03 3.95326317408967009559006555718952e-01 5.97713061106773357877841590379830e-01 3.60171025661803036535535404993880e-03
            6.96062148425968807430308515904471e-03 5.97713061106773357877841590379830e-01 3.95326317408967009559006555718952e-01 3.60171025661803036535535404993880e-03
            2.95682731691414360519765125445701e-02 9.64652531132313861839122637320543e-01 5.77919569854468129221913841320202e-03 9.73861140945065533770785837219819e-04
            2.95682731691414360519765125445701e-02 5.77919569854468129221913841320202e-03 9.64652531132313861839122637320543e-01 9.73861140945065533770785837219819e-04
            9.64652531132313861839122637320543e-01 2.95682731691414360519765125445701e-02 5.77919569854468129221913841320202e-03 9.73861140945065533770785837219819e-04
            9.64652531132313861839122637320543e-01 5.77919569854468129221913841320202e-03 2.95682731691414360519765125445701e-02 9.73861140945065533770785837219819e-04
            5.77919569854468129221913841320202e-03 2.95682731691414360519765125445701e-02 9.64652531132313861839122637320543e-01 9.73861140945065533770785837219819e-04
            5.77919569854468129221913841320202e-03 9.64652531132313861839122637320543e-01 2.95682731691414360519765125445701e-02 9.73861140945065533770785837219819e-04
            3.06750949986907162081450906043756e-01 6.68500829530133566791505472792778e-01 2.47482204829592156158923899056390e-02 5.35413394306625666646537453630117e-03
            3.06750949986907162081450906043756e-01 2.47482204829592156158923899056390e-02 6.68500829530133566791505472792778e-01 5.35413394306625666646537453630117e-03
            6.68500829530133566791505472792778e-01 3.06750949986907162081450906043756e-01 2.47482204829592156158923899056390e-02 5.35413394306625666646537453630117e-03
            6.68500829530133566791505472792778e-01 2.47482204829592156158923899056390e-02 3.06750949986907162081450906043756e-01 5.35413394306625666646537453630117e-03
            2.47482204829592156158923899056390e-02 3.06750949986907162081450906043756e-01 6.68500829530133566791505472792778e-01 5.35413394306625666646537453630117e-03
            2.47482204829592156158923899056390e-02 6.68500829530133566791505472792778e-01 3.06750949986907162081450906043756e-01 5.35413394306625666646537453630117e-03
            2.25171823133371223901733060301922e-01 7.51322633820259988546297336142743e-01 2.35055430463687597963939879264217e-02 4.23736286661477185716551119298856e-03
            2.25171823133371223901733060301922e-01 2.35055430463687597963939879264217e-02 7.51322633820259988546297336142743e-01 4.23736286661477185716551119298856e-03
            7.51322633820259988546297336142743e-01 2.25171823133371223901733060301922e-01 2.35055430463687597963939879264217e-02 4.23736286661477185716551119298856e-03
            7.51322633820259988546297336142743e-01 2.35055430463687597963939879264217e-02 2.25171823133371223901733060301922e-01 4.23736286661477185716551119298856e-03
            2.35055430463687597963939879264217e-02 2.25171823133371223901733060301922e-01 7.51322633820259988546297336142743e-01 4.23736286661477185716551119298856e-03
            2.35055430463687597963939879264217e-02 7.51322633820259988546297336142743e-01 2.25171823133371223901733060301922e-01 4.23736286661477185716551119298856e-03
            7.63589242705955267842554690105317e-02 8.96119411373265895015549631352769e-01 2.75216643561385643224070918222424e-02 3.30847570421702883228243230462340e-03
            7.63589242705955267842554690105317e-02 2.75216643561385643224070918222424e-02 8.96119411373265895015549631352769e-01 3.30847570421702883228243230462340e-03
            8.96119411373265895015549631352769e-01 7.63589242705955267842554690105317e-02 2.75216643561385643224070918222424e-02 3.30847570421702883228243230462340e-03
            8.96119411373265895015549631352769e-01 2.75216643561385643224070918222424e-02 7.63589242705955267842554690105317e-02 3.30847570421702883228243230462340e-03
            2.75216643561385643224070918222424e-02 7.63589242705955267842554690105317e-02 8.96119411373265895015549631352769e-01 3.30847570421702883228243230462340e-03
            2.75216643561385643224070918222424e-02 8.96119411373265895015549631352769e-01 7.63589242705955267842554690105317e-02 3.30847570421702883228243230462340e-03
            1.40424755319720706570052470851806e-01 8.33282278597101022654669577605091e-01 2.62929660831783262864291828009300e-02 4.48583942101860814327674020773884e-03
            1.40424755319720706570052470851806e-01 2.62929660831783262864291828009300e-02 8.33282278597101022654669577605091e-01 4.48583942101860814327674020773884e-03
            8.33282278597101022654669577605091e-01 1.40424755319720706570052470851806e-01 2.62929660831783262864291828009300e-02 4.48583942101860814327674020773884e-03
            8.33282278597101022654669577605091e-01 2.62929660831783262864291828009300e-02 1.40424755319720706570052470851806e-01 4.48583942101860814327674020773884e-03
            2.62929660831783262864291828009300e-02 1.40424755319720706570052470851806e-01 8.33282278597101022654669577605091e-01 4.48583942101860814327674020773884e-03
            2.62929660831783262864291828009300e-02 8.33282278597101022654669577605091e-01 1.40424755319720706570052470851806e-01 4.48583942101860814327674020773884e-03
            3.91187470528715885986770217641606e-01 5.70828988946343374166758621868212e-01 3.79835405249406843353199292323552e-02 7.76555845120818419302644031176897e-03
            3.91187470528715885986770217641606e-01 3.79835405249406843353199292323552e-02 5.70828988946343374166758621868212e-01 7.76555845120818419302644031176897e-03
            5.70828988946343374166758621868212e-01 3.91187470528715885986770217641606e-01 3.79835405249406843353199292323552e-02 7.76555845120818419302644031176897e-03
            5.70828988946343374166758621868212e-01 3.79835405249406843353199292323552e-02 3.91187470528715885986770217641606e-01 7.76555845120818419302644031176897e-03
            3.79835405249406843353199292323552e-02 3.91187470528715885986770217641606e-01 5.70828988946343374166758621868212e-01 7.76555845120818419302644031176897e-03
            3.79835405249406843353199292323552e-02 5.70828988946343374166758621868212e-01 3.91187470528715885986770217641606e-01 7.76555845120818419302644031176897e-03
            1.93960811129872207070334866330086e-01 7.50102714981103257940731054986827e-01 5.59364738890245627445096943120006e-02 6.87043734971176656145530614594463e-03
            1.93960811129872207070334866330086e-01 5.59364738890245627445096943120006e-02 7.50102714981103257940731054986827e-01 6.87043734971176656145530614594463e-03
            7.50102714981103257940731054986827e-01 1.93960811129872207070334866330086e-01 5.59364738890245627445096943120006e-02 6.87043734971176656145530614594463e-03
            7.50102714981103257940731054986827e-01 5.59364738890245627445096943120006e-02 1.93960811129872207070334866330086e-01 6.87043734971176656145530614594463e-03
            5.59364738890245627445096943120006e-02 1.93960811129872207070334866330086e-01 7.50102714981103257940731054986827e-01 6.87043734971176656145530614594463e-03
            5.59364738890245627445096943120006e-02 7.50102714981103257940731054986827e-01 1.93960811129872207070334866330086e-01 6.87043734971176656145530614594463e-03
            1.15860207213215674060258209010499e-01 8.17774906244078891859317081980407e-01 6.63648865427054479582125168235507e-02 5.84068837096601324587741288496545e-03
            1.15860207213215674060258209010499e-01 6.63648865427054479582125168235507e-02 8.17774906244078891859317081980407e-01 5.84068837096601324587741288496545e-03
            8.17774906244078891859317081980407e-01 1.15860207213215674060258209010499e-01 6.63648865427054479582125168235507e-02 5.84068837096601324587741288496545e-03
            8.17774906244078891859317081980407e-01 6.63648865427054479582125168235507e-02 1.15860207213215674060258209010499e-01 5.84068837096601324587741288496545e-03
            6.63648865427054479582125168235507e-02 1.15860207213215674060258209010499e-01 8.17774906244078891859317081980407e-01 5.84068837096601324587741288496545e-03
            6.63648865427054479582125168235507e-02 8.17774906244078891859317081980407e-01 1.15860207213215674060258209010499e-01 5.84068837096601324587741288496545e-03
            2.80237285745932929170720626643742e-01 6.55282643699637845635663779830793e-01 6.44800705544291696824643622676376e-02 8.88657317183859894738073847975102e-03
            2.80237285745932929170720626643742e-01 6.44800705544291696824643622676376e-02 6.55282643699637845635663779830793e-01 8.88657317183859894738073847975102e-03
            6.55282643699637845635663779830793e-01 2.80237285745932929170720626643742e-01 6.44800705544291696824643622676376e-02 8.88657317183859894738073847975102e-03
            6.55282643699637845635663779830793e-01 6.44800705544291696824643622676376e-02 2.80237285745932929170720626643742e-01 8.88657317183859894738073847975102e-03
            6.44800705544291696824643622676376e-02 2.80237285745932929170720626643742e-01 6.55282643699637845635663779830793e-01 8.88657317183859894738073847975102e-03
            6.44800705544291696824643622676376e-02 6.55282643699637845635663779830793e-01 2.80237285745932929170720626643742e-01 8.88657317183859894738073847975102e-03
            3.54893347574873008198892421205528e-01 5.57875630565633384705392927571665e-01 8.72310218594936070957146512228064e-02 1.03831577856246578400734037472830e-02
            3.54893347574873008198892421205528e-01 8.72310218594936070957146512228064e-02 5.57875630565633384705392927571665e-01 1.03831577856246578400734037472830e-02
            5.57875630565633384705392927571665e-01 3.54893347574873008198892421205528e-01 8.72310218594936070957146512228064e-02 1.03831577856246578400734037472830e-02
            5.57875630565633384705392927571665e-01 8.72310218594936070957146512228064e-02 3.54893347574873008198892421205528e-01 1.03831577856246578400734037472830e-02
            8.72310218594936070957146512228064e-02 3.54893347574873008198892421205528e-01 5.57875630565633384705392927571665e-01 1.03831577856246578400734037472830e-02
            8.72310218594936070957146512228064e-02 5.57875630565633384705392927571665e-01 3.54893347574873008198892421205528e-01 1.03831577856246578400734037472830e-02
            1.45342107354678340414011472603306e-01 7.49654646588252315808631465188228e-01 1.05003246057069343777357062208466e-01 6.55128753862388273077055345083863e-03
            1.45342107354678340414011472603306e-01 1.05003246057069343777357062208466e-01 7.49654646588252315808631465188228e-01 6.55128753862388273077055345083863e-03
            7.49654646588252315808631465188228e-01 1.45342107354678340414011472603306e-01 1.05003246057069343777357062208466e-01 6.55128753862388273077055345083863e-03
            7.49654646588252315808631465188228e-01 1.05003246057069343777357062208466e-01 1.45342107354678340414011472603306e-01 6.55128753862388273077055345083863e-03
            1.05003246057069343777357062208466e-01 1.45342107354678340414011472603306e-01 7.49654646588252315808631465188228e-01 6.55128753862388273077055345083863e-03
            1.05003246057069343777357062208466e-01 7.49654646588252315808631465188228e-01 1.45342107354678340414011472603306e-01 6.55128753862388273077055345083863e-03
            2.13535944606159017178370618239569e-01 6.78156200353984717210664712183643e-01 1.08307855039856293366540285205701e-01 8.77076875255828475319574977220327e-03
            2.13535944606159017178370618239569e-01 1.08307855039856293366540285205701e-01 6.78156200353984717210664712183643e-01 8.77076875255828475319574977220327e-03
            6.78156200353984717210664712183643e-01 2.13535944606159017178370618239569e-01 1.08307855039856293366540285205701e-01 8.77076875255828475319574977220327e-03
            6.78156200353984717210664712183643e-01 1.08307855039856293366540285205701e-01 2.13535944606159017178370618239569e-01 8.77076875255828475319574977220327e-03
            1.08307855039856293366540285205701e-01 2.13535944606159017178370618239569e-01 6.78156200353984717210664712183643e-01 8.77076875255828475319574977220327e-03
            1.08307855039856293366540285205701e-01 6.78156200353984717210664712183643e-01 2.13535944606159017178370618239569e-01 8.77076875255828475319574977220327e-03
            3.89059867579780116475518525476218e-01 4.74786102188072933749651838297723e-01 1.36154030232147005285980867483886e-01 1.21889293572907207036104537678511e-02
            3.89059867579780116475518525476218e-01 1.36154030232147005285980867483886e-01 4.74786102188072933749651838297723e-01 1.21889293572907207036104537678511e-02
            4.74786102188072933749651838297723e-01 3.89059867579780116475518525476218e-01 1.36154030232147005285980867483886e-01 1.21889293572907207036104537678511e-02
            4.74786102188072933749651838297723e-01 1.36154030232147005285980867483886e-01 3.89059867579780116475518525476218e-01 1.21889293572907207036104537678511e-02
            1.36154030232147005285980867483886e-01 3.89059867579780116475518525476218e-01 4.74786102188072933749651838297723e-01 1.21889293572907207036104537678511e-02
            1.36154030232147005285980867483886e-01 4.74786102188072933749651838297723e-01 3.89059867579780116475518525476218e-01 1.21889293572907207036104537678511e-02
            2.75628902229600292184130694295163e-01 5.86030082000936203279195524373790e-01 1.38341015769463449025522550073219e-01 1.25064354962984344149523607825358e-02
            2.75628902229600292184130694295163e-01 1.38341015769463449025522550073219e-01 5.86030082000936203279195524373790e-01 1.25064354962984344149523607825358e-02
            5.86030082000936203279195524373790e-01 2.75628902229600292184130694295163e-01 1.38341015769463449025522550073219e-01 1.25064354962984344149523607825358e-02
            5.86030082000936203279195524373790e-01 1.38341015769463449025522550073219e-01 2.75628902229600292184130694295163e-01 1.25064354962984344149523607825358e-02
            1.38341015769463449025522550073219e-01 2.75628902229600292184130694295163e-01 5.86030082000936203279195524373790e-01 1.25064354962984344149523607825358e-02
            1.38341015769463449025522550073219e-01 5.86030082000936203279195524373790e-01 2.75628902229600292184130694295163e-01 1.25064354962984344149523607825358e-02
            3.13779953299047120740539185135276e-01 4.91511463511942858772840736492071e-01 1.94708583189010075997771309630480e-01 1.26984755922826941104686326866613e-02
            3.13779953299047120740539185135276e-01 1.94708583189010075997771309630480e-01 4.91511463511942858772840736492071e-01 1.26984755922826941104686326866613e-02
            4.91511463511942858772840736492071e-01 3.13779953299047120740539185135276e-01 1.94708583189010075997771309630480e-01 1.26984755922826941104686326866613e-02
            4.91511463511942858772840736492071e-01 1.94708583189010075997771309630480e-01 3.13779953299047120740539185135276e-01 1.26984755922826941104686326866613e-02
            1.94708583189010075997771309630480e-01 3.13779953299047120740539185135276e-01 4.91511463511942858772840736492071e-01 1.26984755922826941104686326866613e-02
            1.94708583189010075997771309630480e-01 4.91511463511942858772840736492071e-01 3.13779953299047120740539185135276e-01 1.26984755922826941104686326866613e-02
            ];
        
    case  29
        
        % ALG. DEG.:   29
        % PTS CARD.:  166
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.3e-15
        
        xyw_bar=[
            3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 3.33333333333333314829616256247391e-01 1.39036919724349616878100732719759e-02
            4.99416262867999560803866643254878e-01 4.99416262867999560803866643254878e-01 1.16747426400087839226671349024400e-03 1.19407074555933538986818298610615e-03
            4.99416262867999560803866643254878e-01 1.16747426400087839226671349024400e-03 4.99416262867999560803866643254878e-01 1.19407074555933538986818298610615e-03
            1.16747426400087839226671349024400e-03 4.99416262867999560803866643254878e-01 4.99416262867999560803866643254878e-01 1.19407074555933538986818298610615e-03
            4.89483469832614981331886383486562e-01 4.89483469832614981331886383486562e-01 2.10330603347700373362272330268752e-02 5.58417807814420939088728701449327e-03
            4.89483469832614981331886383486562e-01 2.10330603347700373362272330268752e-02 4.89483469832614981331886383486562e-01 5.58417807814420939088728701449327e-03
            2.10330603347700373362272330268752e-02 4.89483469832614981331886383486562e-01 4.89483469832614981331886383486562e-01 5.58417807814420939088728701449327e-03
            3.83374906469260035013491005884134e-01 3.83374906469260035013491005884134e-01 2.33250187061479929973017988231732e-01 1.40142965005034987602261509209711e-02
            3.83374906469260035013491005884134e-01 2.33250187061479929973017988231732e-01 3.83374906469260035013491005884134e-01 1.40142965005034987602261509209711e-02
            2.33250187061479929973017988231732e-01 3.83374906469260035013491005884134e-01 3.83374906469260035013491005884134e-01 1.40142965005034987602261509209711e-02
            2.86579671246249501059821795934113e-01 2.86579671246249501059821795934113e-01 4.26840657507500997880356408131775e-01 1.41078060470803245268767156517242e-02
            2.86579671246249501059821795934113e-01 4.26840657507500997880356408131775e-01 2.86579671246249501059821795934113e-01 1.41078060470803245268767156517242e-02
            4.26840657507500997880356408131775e-01 2.86579671246249501059821795934113e-01 2.86579671246249501059821795934113e-01 1.41078060470803245268767156517242e-02
            2.36757437980759072315706248446077e-01 2.36757437980759072315706248446077e-01 5.26485124038481799857436271850020e-01 1.34281043157658958042688368550444e-02
            2.36757437980759072315706248446077e-01 5.26485124038481799857436271850020e-01 2.36757437980759072315706248446077e-01 1.34281043157658958042688368550444e-02
            5.26485124038481799857436271850020e-01 2.36757437980759072315706248446077e-01 2.36757437980759072315706248446077e-01 1.34281043157658958042688368550444e-02
            1.84448542625493788760238089707855e-01 1.84448542625493788760238089707855e-01 6.31102914749012366968372589326464e-01 1.21753749760933685508668133934407e-02
            1.84448542625493788760238089707855e-01 6.31102914749012366968372589326464e-01 1.84448542625493788760238089707855e-01 1.21753749760933685508668133934407e-02
            6.31102914749012366968372589326464e-01 1.84448542625493788760238089707855e-01 1.84448542625493788760238089707855e-01 1.21753749760933685508668133934407e-02
            9.59209723834500288530691136656969e-02 9.59209723834500288530691136656969e-02 8.08158055233099914538286157039693e-01 7.22294427143958372300458492532016e-03
            9.59209723834500288530691136656969e-02 8.08158055233099914538286157039693e-01 9.59209723834500288530691136656969e-02 7.22294427143958372300458492532016e-03
            8.08158055233099914538286157039693e-01 9.59209723834500288530691136656969e-02 9.59209723834500288530691136656969e-02 7.22294427143958372300458492532016e-03
            5.73266371488608539319642432019464e-02 5.73266371488608539319642432019464e-02 8.85346725702278236624920282338280e-01 4.43751867960735010137929279494529e-03
            5.73266371488608539319642432019464e-02 8.85346725702278236624920282338280e-01 5.73266371488608539319642432019464e-02 4.43751867960735010137929279494529e-03
            8.85346725702278236624920282338280e-01 5.73266371488608539319642432019464e-02 5.73266371488608539319642432019464e-02 4.43751867960735010137929279494529e-03
            3.60814709934401383478741465182793e-03 3.60814709934401383478741465182793e-03 9.92783705801311922023444367368938e-01 1.97718815192655830541290651325426e-04
            3.60814709934401383478741465182793e-03 9.92783705801311922023444367368938e-01 3.60814709934401383478741465182793e-03 1.97718815192655830541290651325426e-04
            9.92783705801311922023444367368938e-01 3.60814709934401383478741465182793e-03 3.60814709934401383478741465182793e-03 1.97718815192655830541290651325426e-04
            3.24591680276287153361636228510179e-01 6.73712121758946347327423609385733e-01 1.69619796476649931094016210408881e-03 1.29189419129147118996125964685007e-03
            3.24591680276287153361636228510179e-01 1.69619796476649931094016210408881e-03 6.73712121758946347327423609385733e-01 1.29189419129147118996125964685007e-03
            6.73712121758946347327423609385733e-01 3.24591680276287153361636228510179e-01 1.69619796476649931094016210408881e-03 1.29189419129147118996125964685007e-03
            6.73712121758946347327423609385733e-01 1.69619796476649931094016210408881e-03 3.24591680276287153361636228510179e-01 1.29189419129147118996125964685007e-03
            1.69619796476649931094016210408881e-03 3.24591680276287153361636228510179e-01 6.73712121758946347327423609385733e-01 1.29189419129147118996125964685007e-03
            1.69619796476649931094016210408881e-03 6.73712121758946347327423609385733e-01 3.24591680276287153361636228510179e-01 1.29189419129147118996125964685007e-03
            2.05382961103950267833795351180015e-01 7.92855850980322518495313488529064e-01 1.76118791572721367089116029092111e-03 1.12772642035394014305149656962612e-03
            2.05382961103950267833795351180015e-01 1.76118791572721367089116029092111e-03 7.92855850980322518495313488529064e-01 1.12772642035394014305149656962612e-03
            7.92855850980322518495313488529064e-01 2.05382961103950267833795351180015e-01 1.76118791572721367089116029092111e-03 1.12772642035394014305149656962612e-03
            7.92855850980322518495313488529064e-01 1.76118791572721367089116029092111e-03 2.05382961103950267833795351180015e-01 1.12772642035394014305149656962612e-03
            1.76118791572721367089116029092111e-03 2.05382961103950267833795351180015e-01 7.92855850980322518495313488529064e-01 1.12772642035394014305149656962612e-03
            1.76118791572721367089116029092111e-03 7.92855850980322518495313488529064e-01 2.05382961103950267833795351180015e-01 1.12772642035394014305149656962612e-03
            1.27817155048788905213896782697702e-01 8.66822611309806201340677489497466e-01 5.36023364140492120100134343374521e-03 1.84745171735374602052515147221357e-03
            1.27817155048788905213896782697702e-01 5.36023364140492120100134343374521e-03 8.66822611309806201340677489497466e-01 1.84745171735374602052515147221357e-03
            8.66822611309806201340677489497466e-01 1.27817155048788905213896782697702e-01 5.36023364140492120100134343374521e-03 1.84745171735374602052515147221357e-03
            8.66822611309806201340677489497466e-01 5.36023364140492120100134343374521e-03 1.27817155048788905213896782697702e-01 1.84745171735374602052515147221357e-03
            5.36023364140492120100134343374521e-03 1.27817155048788905213896782697702e-01 8.66822611309806201340677489497466e-01 1.84745171735374602052515147221357e-03
            5.36023364140492120100134343374521e-03 8.66822611309806201340677489497466e-01 1.27817155048788905213896782697702e-01 1.84745171735374602052515147221357e-03
            6.39721317044320281208769074510201e-02 9.34220030168086745803179837821517e-01 1.80783812748119832036763909854926e-03 7.17822033246548430948652530503296e-04
            6.39721317044320281208769074510201e-02 1.80783812748119832036763909854926e-03 9.34220030168086745803179837821517e-01 7.17822033246548430948652530503296e-04
            9.34220030168086745803179837821517e-01 6.39721317044320281208769074510201e-02 1.80783812748119832036763909854926e-03 7.17822033246548430948652530503296e-04
            9.34220030168086745803179837821517e-01 1.80783812748119832036763909854926e-03 6.39721317044320281208769074510201e-02 7.17822033246548430948652530503296e-04
            1.80783812748119832036763909854926e-03 6.39721317044320281208769074510201e-02 9.34220030168086745803179837821517e-01 7.17822033246548430948652530503296e-04
            1.80783812748119832036763909854926e-03 9.34220030168086745803179837821517e-01 6.39721317044320281208769074510201e-02 7.17822033246548430948652530503296e-04
            4.08772703046101637536224870927981e-01 5.83304257472950338225814448378514e-01 7.92303948094796872680944943567738e-03 3.51318664214702713194382610595312e-03
            4.08772703046101637536224870927981e-01 7.92303948094796872680944943567738e-03 5.83304257472950338225814448378514e-01 3.51318664214702713194382610595312e-03
            5.83304257472950338225814448378514e-01 4.08772703046101637536224870927981e-01 7.92303948094796872680944943567738e-03 3.51318664214702713194382610595312e-03
            5.83304257472950338225814448378514e-01 7.92303948094796872680944943567738e-03 4.08772703046101637536224870927981e-01 3.51318664214702713194382610595312e-03
            7.92303948094796872680944943567738e-03 4.08772703046101637536224870927981e-01 5.83304257472950338225814448378514e-01 3.51318664214702713194382610595312e-03
            7.92303948094796872680944943567738e-03 5.83304257472950338225814448378514e-01 4.08772703046101637536224870927981e-01 3.51318664214702713194382610595312e-03
            2.60003660015059478993748598441016e-01 7.28202443651907782573573513218435e-01 1.17938963330327384326778883405495e-02 3.49660600844853850302507147773667e-03
            2.60003660015059478993748598441016e-01 1.17938963330327384326778883405495e-02 7.28202443651907782573573513218435e-01 3.49660600844853850302507147773667e-03
            7.28202443651907782573573513218435e-01 2.60003660015059478993748598441016e-01 1.17938963330327384326778883405495e-02 3.49660600844853850302507147773667e-03
            7.28202443651907782573573513218435e-01 1.17938963330327384326778883405495e-02 2.60003660015059478993748598441016e-01 3.49660600844853850302507147773667e-03
            1.17938963330327384326778883405495e-02 2.60003660015059478993748598441016e-01 7.28202443651907782573573513218435e-01 3.49660600844853850302507147773667e-03
            1.17938963330327384326778883405495e-02 7.28202443651907782573573513218435e-01 2.60003660015059478993748598441016e-01 3.49660600844853850302507147773667e-03
            8.37010376047120779885446495427459e-02 8.98464666406439738111089354788419e-01 1.78342959888482255337294191122055e-02 2.41690948038280955731393895291603e-03
            8.37010376047120779885446495427459e-02 1.78342959888482255337294191122055e-02 8.98464666406439738111089354788419e-01 2.41690948038280955731393895291603e-03
            8.98464666406439738111089354788419e-01 8.37010376047120779885446495427459e-02 1.78342959888482255337294191122055e-02 2.41690948038280955731393895291603e-03
            8.98464666406439738111089354788419e-01 1.78342959888482255337294191122055e-02 8.37010376047120779885446495427459e-02 2.41690948038280955731393895291603e-03
            1.78342959888482255337294191122055e-02 8.37010376047120779885446495427459e-02 8.98464666406439738111089354788419e-01 2.41690948038280955731393895291603e-03
            1.78342959888482255337294191122055e-02 8.98464666406439738111089354788419e-01 8.37010376047120779885446495427459e-02 2.41690948038280955731393895291603e-03
            2.27772424394371494904554253935203e-02 9.71586291829570680711469776724698e-01 5.63646573099219061475650960346684e-03 8.13134666139964990862198934706839e-04
            2.27772424394371494904554253935203e-02 5.63646573099219061475650960346684e-03 9.71586291829570680711469776724698e-01 8.13134666139964990862198934706839e-04
            9.71586291829570680711469776724698e-01 2.27772424394371494904554253935203e-02 5.63646573099219061475650960346684e-03 8.13134666139964990862198934706839e-04
            9.71586291829570680711469776724698e-01 5.63646573099219061475650960346684e-03 2.27772424394371494904554253935203e-02 8.13134666139964990862198934706839e-04
            5.63646573099219061475650960346684e-03 2.27772424394371494904554253935203e-02 9.71586291829570680711469776724698e-01 8.13134666139964990862198934706839e-04
            5.63646573099219061475650960346684e-03 9.71586291829570680711469776724698e-01 2.27772424394371494904554253935203e-02 8.13134666139964990862198934706839e-04
            3.32497033049334811405373102388694e-01 6.42488243894240529385797344730236e-01 2.50147230564246036976783216232434e-02 5.64942974660397490699326183971607e-03
            3.32497033049334811405373102388694e-01 2.50147230564246036976783216232434e-02 6.42488243894240529385797344730236e-01 5.64942974660397490699326183971607e-03
            6.42488243894240529385797344730236e-01 3.32497033049334811405373102388694e-01 2.50147230564246036976783216232434e-02 5.64942974660397490699326183971607e-03
            6.42488243894240529385797344730236e-01 2.50147230564246036976783216232434e-02 3.32497033049334811405373102388694e-01 5.64942974660397490699326183971607e-03
            2.50147230564246036976783216232434e-02 3.32497033049334811405373102388694e-01 6.42488243894240529385797344730236e-01 5.64942974660397490699326183971607e-03
            2.50147230564246036976783216232434e-02 6.42488243894240529385797344730236e-01 3.32497033049334811405373102388694e-01 5.64942974660397490699326183971607e-03
            1.75467715977808663696535518283781e-01 8.03433175906504426500021054380341e-01 2.10991081156869375590190429647919e-02 4.45904031659884441535224297581408e-03
            1.75467715977808663696535518283781e-01 2.10991081156869375590190429647919e-02 8.03433175906504426500021054380341e-01 4.45904031659884441535224297581408e-03
            8.03433175906504426500021054380341e-01 1.75467715977808663696535518283781e-01 2.10991081156869375590190429647919e-02 4.45904031659884441535224297581408e-03
            8.03433175906504426500021054380341e-01 2.10991081156869375590190429647919e-02 1.75467715977808663696535518283781e-01 4.45904031659884441535224297581408e-03
            2.10991081156869375590190429647919e-02 1.75467715977808663696535518283781e-01 8.03433175906504426500021054380341e-01 4.45904031659884441535224297581408e-03
            2.10991081156869375590190429647919e-02 8.03433175906504426500021054380341e-01 1.75467715977808663696535518283781e-01 4.45904031659884441535224297581408e-03
            4.23914433713915364543289854282193e-02 9.37282237421026453993988525326131e-01 2.03263192075819887350007775239646e-02 1.85640152529499774872179251872240e-03
            4.23914433713915364543289854282193e-02 2.03263192075819887350007775239646e-02 9.37282237421026453993988525326131e-01 1.85640152529499774872179251872240e-03
            9.37282237421026453993988525326131e-01 4.23914433713915364543289854282193e-02 2.03263192075819887350007775239646e-02 1.85640152529499774872179251872240e-03
            9.37282237421026453993988525326131e-01 2.03263192075819887350007775239646e-02 4.23914433713915364543289854282193e-02 1.85640152529499774872179251872240e-03
            2.03263192075819887350007775239646e-02 4.23914433713915364543289854282193e-02 9.37282237421026453993988525326131e-01 1.85640152529499774872179251872240e-03
            2.03263192075819887350007775239646e-02 9.37282237421026453993988525326131e-01 4.23914433713915364543289854282193e-02 1.85640152529499774872179251872240e-03
            2.38086960749264359149535152937460e-01 7.16981908850933602472821348783327e-01 4.49311303998020106220678826502990e-02 7.01742231395961670731287185276415e-03
            2.38086960749264359149535152937460e-01 4.49311303998020106220678826502990e-02 7.16981908850933602472821348783327e-01 7.01742231395961670731287185276415e-03
            7.16981908850933602472821348783327e-01 2.38086960749264359149535152937460e-01 4.49311303998020106220678826502990e-02 7.01742231395961670731287185276415e-03
            7.16981908850933602472821348783327e-01 4.49311303998020106220678826502990e-02 2.38086960749264359149535152937460e-01 7.01742231395961670731287185276415e-03
            4.49311303998020106220678826502990e-02 2.38086960749264359149535152937460e-01 7.16981908850933602472821348783327e-01 7.01742231395961670731287185276415e-03
            4.49311303998020106220678826502990e-02 7.16981908850933602472821348783327e-01 2.38086960749264359149535152937460e-01 7.01742231395961670731287185276415e-03
            1.13850259493195438142976172457566e-01 8.44311313344634828403911797067849e-01 4.18384271621697889642632617324125e-02 4.76963067992195680760980636136992e-03
            1.13850259493195438142976172457566e-01 4.18384271621697889642632617324125e-02 8.44311313344634828403911797067849e-01 4.76963067992195680760980636136992e-03
            8.44311313344634828403911797067849e-01 1.13850259493195438142976172457566e-01 4.18384271621697889642632617324125e-02 4.76963067992195680760980636136992e-03
            8.44311313344634828403911797067849e-01 4.18384271621697889642632617324125e-02 1.13850259493195438142976172457566e-01 4.76963067992195680760980636136992e-03
            4.18384271621697889642632617324125e-02 1.13850259493195438142976172457566e-01 8.44311313344634828403911797067849e-01 4.76963067992195680760980636136992e-03
            4.18384271621697889642632617324125e-02 8.44311313344634828403911797067849e-01 1.13850259493195438142976172457566e-01 4.76963067992195680760980636136992e-03
            4.07699471210733355164279601012822e-01 5.48247215417901112566312349372311e-01 4.40533133713655322694080496148672e-02 7.68181583252763606345370206440748e-03
            4.07699471210733355164279601012822e-01 4.40533133713655322694080496148672e-02 5.48247215417901112566312349372311e-01 7.68181583252763606345370206440748e-03
            5.48247215417901112566312349372311e-01 4.07699471210733355164279601012822e-01 4.40533133713655322694080496148672e-02 7.68181583252763606345370206440748e-03
            5.48247215417901112566312349372311e-01 4.40533133713655322694080496148672e-02 4.07699471210733355164279601012822e-01 7.68181583252763606345370206440748e-03
            4.40533133713655322694080496148672e-02 4.07699471210733355164279601012822e-01 5.48247215417901112566312349372311e-01 7.68181583252763606345370206440748e-03
            4.40533133713655322694080496148672e-02 5.48247215417901112566312349372311e-01 4.07699471210733355164279601012822e-01 7.68181583252763606345370206440748e-03
            3.09719966010297276248763864714419e-01 6.22325578952138247501579826348461e-01 6.79544550375644762496563089371193e-02 8.66304121915657115993170123147138e-03
            3.09719966010297276248763864714419e-01 6.79544550375644762496563089371193e-02 6.22325578952138247501579826348461e-01 8.66304121915657115993170123147138e-03
            6.22325578952138247501579826348461e-01 3.09719966010297276248763864714419e-01 6.79544550375644762496563089371193e-02 8.66304121915657115993170123147138e-03
            6.22325578952138247501579826348461e-01 6.79544550375644762496563089371193e-02 3.09719966010297276248763864714419e-01 8.66304121915657115993170123147138e-03
            6.79544550375644762496563089371193e-02 3.09719966010297276248763864714419e-01 6.22325578952138247501579826348461e-01 8.66304121915657115993170123147138e-03
            6.79544550375644762496563089371193e-02 6.22325578952138247501579826348461e-01 3.09719966010297276248763864714419e-01 8.66304121915657115993170123147138e-03
            1.65012577073424038243132372372202e-01 7.67767929108697733120436623721616e-01 6.72194938178781731252797726483550e-02 6.94192073189619281853302013018947e-03
            1.65012577073424038243132372372202e-01 6.72194938178781731252797726483550e-02 7.67767929108697733120436623721616e-01 6.94192073189619281853302013018947e-03
            7.67767929108697733120436623721616e-01 1.65012577073424038243132372372202e-01 6.72194938178781731252797726483550e-02 6.94192073189619281853302013018947e-03
            7.67767929108697733120436623721616e-01 6.72194938178781731252797726483550e-02 1.65012577073424038243132372372202e-01 6.94192073189619281853302013018947e-03
            6.72194938178781731252797726483550e-02 1.65012577073424038243132372372202e-01 7.67767929108697733120436623721616e-01 6.94192073189619281853302013018947e-03
            6.72194938178781731252797726483550e-02 7.67767929108697733120436623721616e-01 1.65012577073424038243132372372202e-01 6.94192073189619281853302013018947e-03
            4.37364893469387783220270193851320e-01 4.84688748698669014114415176663897e-01 7.79463578319432026653146294847829e-02 6.08713825993679926579238426143093e-03
            4.37364893469387783220270193851320e-01 7.79463578319432026653146294847829e-02 4.84688748698669014114415176663897e-01 6.08713825993679926579238426143093e-03
            4.84688748698669014114415176663897e-01 4.37364893469387783220270193851320e-01 7.79463578319432026653146294847829e-02 6.08713825993679926579238426143093e-03
            4.84688748698669014114415176663897e-01 7.79463578319432026653146294847829e-02 4.37364893469387783220270193851320e-01 6.08713825993679926579238426143093e-03
            7.79463578319432026653146294847829e-02 4.37364893469387783220270193851320e-01 4.84688748698669014114415176663897e-01 6.08713825993679926579238426143093e-03
            7.79463578319432026653146294847829e-02 4.84688748698669014114415176663897e-01 4.37364893469387783220270193851320e-01 6.08713825993679926579238426143093e-03
            2.30361692319617394897335316272802e-01 6.66275923025302252433732519421028e-01 1.03362384655080297157780933048343e-01 9.87867211691080895907823844481754e-03
            2.30361692319617394897335316272802e-01 1.03362384655080297157780933048343e-01 6.66275923025302252433732519421028e-01 9.87867211691080895907823844481754e-03
            6.66275923025302252433732519421028e-01 2.30361692319617394897335316272802e-01 1.03362384655080297157780933048343e-01 9.87867211691080895907823844481754e-03
            6.66275923025302252433732519421028e-01 1.03362384655080297157780933048343e-01 2.30361692319617394897335316272802e-01 9.87867211691080895907823844481754e-03
            1.03362384655080297157780933048343e-01 2.30361692319617394897335316272802e-01 6.66275923025302252433732519421028e-01 9.87867211691080895907823844481754e-03
            1.03362384655080297157780933048343e-01 6.66275923025302252433732519421028e-01 2.30361692319617394897335316272802e-01 9.87867211691080895907823844481754e-03
            3.53054767022387583352838191785850e-01 5.40538992741738999470157978066709e-01 1.06406240235873417177003830147441e-01 1.04031774001664811118983422488782e-02
            3.53054767022387583352838191785850e-01 1.06406240235873417177003830147441e-01 5.40538992741738999470157978066709e-01 1.04031774001664811118983422488782e-02
            5.40538992741738999470157978066709e-01 3.53054767022387583352838191785850e-01 1.06406240235873417177003830147441e-01 1.04031774001664811118983422488782e-02
            5.40538992741738999470157978066709e-01 1.06406240235873417177003830147441e-01 3.53054767022387583352838191785850e-01 1.04031774001664811118983422488782e-02
            1.06406240235873417177003830147441e-01 3.53054767022387583352838191785850e-01 5.40538992741738999470157978066709e-01 1.04031774001664811118983422488782e-02
            1.06406240235873417177003830147441e-01 5.40538992741738999470157978066709e-01 3.53054767022387583352838191785850e-01 1.04031774001664811118983422488782e-02
            1.58718755804575895052010992003488e-01 7.21198858207568349598659551702440e-01 1.20082385987855699838178225036245e-01 6.77295750139823762164814624497922e-03
            1.58718755804575895052010992003488e-01 1.20082385987855699838178225036245e-01 7.21198858207568349598659551702440e-01 6.77295750139823762164814624497922e-03
            7.21198858207568349598659551702440e-01 1.58718755804575895052010992003488e-01 1.20082385987855699838178225036245e-01 6.77295750139823762164814624497922e-03
            7.21198858207568349598659551702440e-01 1.20082385987855699838178225036245e-01 1.58718755804575895052010992003488e-01 6.77295750139823762164814624497922e-03
            1.20082385987855699838178225036245e-01 1.58718755804575895052010992003488e-01 7.21198858207568349598659551702440e-01 6.77295750139823762164814624497922e-03
            1.20082385987855699838178225036245e-01 7.21198858207568349598659551702440e-01 1.58718755804575895052010992003488e-01 6.77295750139823762164814624497922e-03
            4.02149591942053019621994280896615e-01 4.54827385300507591736618451250251e-01 1.43023022757439388641387267853133e-01 8.32356466794720653568084856033238e-03
            4.02149591942053019621994280896615e-01 1.43023022757439388641387267853133e-01 4.54827385300507591736618451250251e-01 8.32356466794720653568084856033238e-03
            4.54827385300507591736618451250251e-01 4.02149591942053019621994280896615e-01 1.43023022757439388641387267853133e-01 8.32356466794720653568084856033238e-03
            4.54827385300507591736618451250251e-01 1.43023022757439388641387267853133e-01 4.02149591942053019621994280896615e-01 8.32356466794720653568084856033238e-03
            1.43023022757439388641387267853133e-01 4.02149591942053019621994280896615e-01 4.54827385300507591736618451250251e-01 8.32356466794720653568084856033238e-03
            1.43023022757439388641387267853133e-01 4.54827385300507591736618451250251e-01 4.02149591942053019621994280896615e-01 8.32356466794720653568084856033238e-03
            2.71868486061960623612066001442145e-01 5.77168860612755296735087995330105e-01 1.50962653325284024141694771969924e-01 1.17582567855695676456440068591291e-02
            2.71868486061960623612066001442145e-01 1.50962653325284024141694771969924e-01 5.77168860612755296735087995330105e-01 1.17582567855695676456440068591291e-02
            5.77168860612755296735087995330105e-01 2.71868486061960623612066001442145e-01 1.50962653325284024141694771969924e-01 1.17582567855695676456440068591291e-02
            5.77168860612755296735087995330105e-01 1.50962653325284024141694771969924e-01 2.71868486061960623612066001442145e-01 1.17582567855695676456440068591291e-02
            1.50962653325284024141694771969924e-01 2.71868486061960623612066001442145e-01 5.77168860612755296735087995330105e-01 1.17582567855695676456440068591291e-02
            1.50962653325284024141694771969924e-01 5.77168860612755296735087995330105e-01 2.71868486061960623612066001442145e-01 1.17582567855695676456440068591291e-02
            3.28703899955314460257227437978145e-01 4.78814632497088499274440209774184e-01 1.92481467547597040468332352247671e-01 1.26811781993145661351363173707796e-02
            3.28703899955314460257227437978145e-01 1.92481467547597040468332352247671e-01 4.78814632497088499274440209774184e-01 1.26811781993145661351363173707796e-02
            4.78814632497088499274440209774184e-01 3.28703899955314460257227437978145e-01 1.92481467547597040468332352247671e-01 1.26811781993145661351363173707796e-02
            4.78814632497088499274440209774184e-01 1.92481467547597040468332352247671e-01 3.28703899955314460257227437978145e-01 1.26811781993145661351363173707796e-02
            1.92481467547597040468332352247671e-01 3.28703899955314460257227437978145e-01 4.78814632497088499274440209774184e-01 1.26811781993145661351363173707796e-02
            1.92481467547597040468332352247671e-01 4.78814632497088499274440209774184e-01 3.28703899955314460257227437978145e-01 1.26811781993145661351363173707796e-02
            ];
        
        
    otherwise
        xyw=[];
end




pointset_stats=[
    1    1 1.92e-16    0    0
    2    3 3.68e-16    0    0
    3    6 1.46e-16    0    0
    4    6 1.39e-16    0    0
    5    7 1.98e-16    0    0
    6   12 2.36e-16    0    0
    7   16 3.75e-16    0    0
    8   21 4.42e-13    0    0
    9   19 5.55e-16    0    0
    10   25 4.72e-16    0    0
    11   28 8.33e-16    0    0
    12   33 4.82e-16    0    0
    13   37 5.83e-16    0    0
    14   42 4.58e-16    0    0
    15   49 4.82e-16    0    0
    16   55 8.12e-16    0    0
    17   60 6.21e-16    0    0
    18   73 1.01e-15    0    0
    20   82 9.06e-16    0    0
    21   87 5.59e-16    0    0
    22   96 6.66e-16    0    0
    24  112 1.34e-15    0    0
    25  126 6.17e-16    0    0
    26  133 8.50e-16    0    0
    27  145 9.99e-16    0    0
    28  154 1.11e-15    0    0
    29  166 1.33e-15    0    0
    ];





