
function [S1,S2,S3,pointset_stats]=set_zhang_cui_liu_compact(deg)

%--------------------------------------------------------------------
% CUBATURE RULE ON THE SIMPLEX [0 0; 0 1; 1 0] WITH ADE EQUAL TO deg.
% THE CUBATURE RULE IS DESCRIBED IN COMPACT FORM VIA ITS ROTATIONS.
%--------------------------------------------------------------------
% INPUT:
% deg: algebraic degree of precision (ADE) of the rule.
%--------------------------------------------------------------------
% OUTPUT:
% S1,S2,S3,w:
%    S1 is a matrix N1 x 3,
%    S2 is a matrix N2 x 2,
%    S3 is a matrix N3 x 1,
%    with N1, N2, N3 possibly zero.
%    The pointset is written in terms of rotations of type s1, s2, s3
%    applied resp. to S1, S2, S3, where:
%    s1(w,a,b): [a b c w; a c b w; b a c w; b c a w; c a b w; c b a w];
%            with c=1-a-b.
%    s2(w,a): [a a c w; a c a w; c a a w];
%            with c=1-2*a.
%    s3(w): [1/3 1/3 1/3 w];
%    S1 gives points of which one has to consider all the rotations of
%    type s1.
%    S2 gives points of which one has to consider all the rotations of
%    type s2.
%    S3 gives points of which one has to consider all the rotations of
%    type s3.
%    REMARK: the sum of the weights of the set extracted from S1,S2,S3,
%            for each rule, makes 1.
%    REMARK: in S1, S2, S3, the first component is that of the weight w.
%
% pointset_stats:
%    statistics of the pointset stored in a matrix. Each rows contains
%    degree, cardinality of the full pointset, moment error w.r.t. orthn.
%    Dubiner basis, number of points with negative weights, number of
%    points outside the domain.
%--------------------------------------------------------------------
% SOURCE:
% L. Zhang, T. Cui, H. Liu
% "A set of Symmetric Quadrature rules on triangles and tetrahedra"
% J. Comput. Math. 27, No. I, (2009), pp.89-96.
%--------------------------------------------------------------------
% | DEG | CARD | W- | PT.O|   M.E.   |Q.|
%--------------------------------------------------------------------
% |   1 |   1 |   0 |   0 | 1.92e-16 |PI|
% |   2 |   3 |   0 |   0 | 3.68e-16 |PI|
% |   3 |   6 |   0 |   0 | 1.46e-16 |PI|
% |   4 |   6 |   0 |   0 | 1.39e-16 |PI|
% |   5 |   7 |   0 |   0 | 1.98e-16 |PI|
% |   6 |  12 |   0 |   0 | 2.36e-16 |PI|
% |   7 |  16 |   0 |   0 | 3.75e-16 |PI|
% |   8 |  21 |   0 |   0 | 4.42e-13 |PI|
% |   9 |  19 |   0 |   0 | 5.55e-16 |PI|
% |  10 |  25 |   0 |   0 | 4.72e-16 |PI|
% |  11 |  28 |   0 |   0 | 8.33e-16 |PI|
% |  12 |  33 |   0 |   0 | 4.82e-16 |PI|
% |  13 |  37 |   0 |   0 | 5.83e-16 |PI|
% |  14 |  42 |   0 |   0 | 4.58e-16 |PI|
% |  15 |  49 |   0 |   0 | 4.82e-16 |PI|
% |  16 |  55 |   0 |   0 | 8.12e-16 |PI|
% |  17 |  60 |   0 |   0 | 6.21e-16 |PI|
% |  18 |  73 |   0 |   0 | 1.01e-15 |PI|
% |  20 |  82 |   0 |   0 | 9.06e-16 |PI|
% |  21 |  87 |   0 |   0 | 5.59e-16 |PI|
% |  22 |  96 |   0 |   0 | 6.66e-16 |PI|
% |  24 | 112 |   0 |   0 | 1.34e-15 |PI|
% |  25 | 126 |   0 |   0 | 6.17e-16 |PI|
% |  26 | 133 |   0 |   0 | 8.50e-16 |PI|
% |  27 | 145 |   0 |   0 | 9.99e-16 |PI|
% |  28 | 154 |   0 |   0 | 1.11e-15 |PI|
% |  29 | 166 |   0 |   0 | 1.33e-15 |PI|
%--------------------------------------------------------------------
% LEGEND:
%--------------------------------------------------------------------
% DEG  : algebraic degree of precision
% CARD : cardinality of the pointset
% W-   : negative weights
% PT.O : points outside the simplex [0 0; 1 0; 0 1].
% M.E. : maximum moment error in inf norm.
% Q.: quality of the rule.
%        PI: pos. weights, points in simplex
%        NI: some neg. weights, points in simplex
%        P0: pos. weights, some points out of the simplex
%        N0: some neg. weights, some points out of the simplex
%--------------------------------------------------------------------


switch deg
    
    case   1
        
        % ALG. DEG.:    1
        % PTS CARD.:    1
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.9e-16
        
        S1=[];
        
        S2=[];
        
        S3=[
            1.00000000000000000000000000000000e+00
            ];
        
        
    case   2
        
        % ALG. DEG.:    2
        % PTS CARD.:    3
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.7e-16
        
        S1=[];
        
        S2=[
            3.33333333333333314829616256247391e-01 1.66666666666666657414808128123695e-01
            ];
        
        S3=[];
        
        
    case   3
        
        % ALG. DEG.:    3
        % PTS CARD.:    6
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.5e-16
        
        S1=[];
        
        S2=[
            2.81149802440979645901109051919775e-01 1.62882850395891898287459298444446e-01
            5.21835308923536828062950121420727e-02 4.77919883567563708037795322525199e-01
            ];
        
        S3=[];
        
        
    case   4
        
        % ALG. DEG.:    4
        % PTS CARD.:    6
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.4e-16
        
        S1=[];
        
        S2=[
            2.23381589678011471811203136894619e-01 4.45948490915964890213274429697776e-01
            1.09951743655321870773988734981685e-01 9.15762135097707430375635340169538e-02
            ];
        
        S3=[];
        
        
    case   5
        
        % ALG. DEG.:    5
        % PTS CARD.:    7
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 2.0e-16
        
        S1=[];
        
        S2=[
            1.25939180544827139529573400977824e-01 1.01286507323456342888334802410100e-01
            1.32394152788506191953388224646915e-01 4.70142064105115109473587153843255e-01
            ];
        
        S3=[
            2.25000000000000005551115123125783e-01
            ];
        
        
    case   6
        
        % ALG. DEG.:    6
        % PTS CARD.:   12
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 2.4e-16
        
        S1=[
            8.28510756183735708191306912340224e-02 5.31450498448169453280520713178703e-02 3.10352451033784393352732422499685e-01
            ];
        
        S2=[
            5.08449063702068188019822514434054e-02 6.30890144915022266225435032538371e-02
            1.16786275726379368267160430150398e-01 2.49286745170910428726074314909056e-01
            ];
        
        S3=[];
        
        
    case   7
        
        % ALG. DEG.:    7
        % PTS CARD.:   16
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 3.7e-16
        
        S1=[
            2.72303141744349927466650740370824e-02 2.63112829634638112352718053443823e-01 8.39477740995760515763102205255564e-03
            ];
        
        S2=[
            1.03217370534718244634575512463925e-01 1.70569307751760212976677166807349e-01
            3.24584976231980792960030157701112e-02 5.05472283170309774624762155781355e-02
            9.50916342672846193195823616406415e-02 4.59292588292723180654064663031022e-01
            ];
        
        S3=[
            1.44315607677787172136163462710101e-01
            ];
        
        
    case   8
        
        % ALG. DEG.:    8
        % PTS CARD.:   21
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.4e-13
        
        S1=[
            7.07034101783999974211170069793297e-02 7.47512472733899979715488370857202e-01 2.22063165537299989082953288743738e-01
            9.09390760951999943539902915290440e-02 1.36991201264899997713797574760974e-01 2.18290070971399990940042812326283e-01
            ];
        
        S2=[
            5.19871420645999990362007281419210e-02 4.51890097843999985482987824525480e-02
            1.03234405138000001622700096959306e-01 4.81519834783300015867979482209194e-01
            1.88160146916699994035226950472861e-01 4.03603979817900027082799851996242e-01
            ];
        
        S3=[];
        
        
    case   9
        
        % ALG. DEG.:    9
        % PTS CARD.:   19
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.6e-16
        
        S1=[
            4.32835393772893761710207627402269e-02 7.41198598784498008384957756788936e-01 2.21962989160765705731748198559217e-01
            ];
        
        S2=[
            3.13347002271390714933652077434090e-02 4.89682519198737620236272505280795e-01
            2.55776756586980312524470804191878e-02 4.47295133944527120717360446633393e-02
            7.78275410047742782770896496913338e-02 4.37089591492936635486188379218220e-01
            7.96477389272102487272064763601520e-02 1.88203535619032719106158424438036e-01
            ];
        
        S3=[
            9.71357962827988363763154211483197e-02
            ];
        
        
    case  10
        
        % ALG. DEG.:   10
        % PTS CARD.:   25
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.7e-16
        
        S1=[
            3.74688582104676440320822905505338e-02 6.54268667920066104848331178800436e-01 3.08046001685247705648151850255090e-01
            2.69493525918799610441300984575719e-02 1.22804577068559267472025453571405e-01 3.33718337393047875805152102657303e-02
            ];
        
        S2=[
            7.72985880029631172716264586597390e-02 4.27273178846775514294620279542869e-01
            7.84576386123717328668902837307542e-02 1.83099222448675019725428114725219e-01
            1.74691679959294854374007854858064e-02 4.90434019701130596846638809438446e-01
            4.29237418483282770487186041918903e-03 1.25724455515805327854295114775596e-02
            ];
        
        S3=[
            8.09374287976228762309105491112859e-02
            ];
        
        
    case  11
        
        % ALG. DEG.:   11
        % PTS CARD.:   28
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 8.3e-16
        
        S1=[
            1.47622727177161017220186067788745e-02 8.25618766164862849521455245849211e-01 1.59742304591850181605394709549728e-01
            4.07279964582990400279172149566875e-02 6.40472310134865274733328988077119e-01 3.11783715709599007492869304769556e-01
            ];
        
        S2=[
            1.23240435069094933506761435637600e-02 3.09383552454307840517966354809687e-02
            6.28280097444101071069155750592472e-02 4.36498181134128859781640130677260e-01
            1.22203790493645297887681522297498e-02 4.98984763702593259182549445540644e-01
            6.77013489528115047777490076441609e-02 2.14688197958594340075677564527723e-01
            4.02196936288516937452186539303511e-02 1.13683104042113389264834211189736e-01
            ];
        
        S3=[
            8.11779602968671631924024723048205e-02
            ];
        
        
    case  12
        
        % ALG. DEG.:   12
        % PTS CARD.:   33
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.8e-16
        
        S1=[
            2.23567732023034447530474011500701e-02 6.95836086787803398401308641041396e-01 2.81325580989939538767430349253118e-01
            1.73162311086588922848239491258937e-02 8.58014033544072685089076912845485e-01 1.16251915907597147548102611835930e-01
            4.03715577663809263575522834344156e-02 6.08943235779787817030239693849580e-01 2.75713269685514184015318051024224e-01
            ];
        
        S2=[
            6.16626105155901760995762472816750e-03 2.13173504532103708086765436746646e-02
            6.28582242178851030578456970943080e-02 2.71210385012115917646724483347498e-01
            3.47961129307089445106271341501269e-02 1.27576145541585916909355091775069e-01
            4.36925445380384053217781570310763e-02 4.39724392294460253438614927290473e-01
            2.57310664404553357531568735794281e-02 4.88217389773804855135352909201174e-01
            ];
        
        S3=[];
        
        
    case  13
        
        % ALG. DEG.:   13
        % PTS CARD.:   37
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.8e-16
        
        S1=[
            3.46412761408483732550322997667536e-02 6.23545995553675624911704744590679e-01 3.08441760892117777803633771327441e-01
            1.49654011051656675446164257436976e-02 8.64707770295442768038185477053048e-01 1.10922042803463391513751901129581e-01
            2.41790398115938193202012485016894e-02 7.48507115899952224502555964136263e-01 1.63597401067850478639797984214965e-01
            9.59068100354326311118846604131249e-03 7.22357793124188019007192451681476e-01 2.72515817773429647186134161529480e-01
            ];
        
        S2=[
            5.56019675304533292359288054740318e-02 4.26941414259800422481561099630198e-01
            5.82784851191999811459432123683655e-02 2.21372286291832892635156326832657e-01
            6.05233710353917167545922950466775e-03 2.15096811088431837111212274749050e-02
            2.39944019288947305024173317633540e-02 4.89076946452539351728461269885884e-01
            ];
        
        S3=[
            6.79600365868316402817583821160952e-02
            ];
        
        
    case  14
        
        % ALG. DEG.:   14
        % PTS CARD.:   42
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.6e-16
        
        S1=[
            2.46657532125636735909068875116645e-02 1.72266687821355568566161764465505e-01 5.71247574036479396908383421305189e-02
            3.85715107870606838114113656956761e-02 5.70222290846683188547672216373030e-01 9.29162493569718195285389583659708e-02
            1.44363081135338404414891755322969e-02 2.98372882136257733254325330563006e-01 1.46469500556544104941103512373957e-02
            5.01022883850067192457178322229083e-03 1.18974497696956851844340974366787e-01 1.26833093287202509219513402172197e-03
            ];
        
        S2=[
            4.21625887369930160697428789262631e-02 1.77205532412543442788432912493590e-01
            4.92340360240008188125182542194125e-03 1.93909612487010475645465845673243e-02
            1.44336996697766676428509313723225e-02 6.17998830908726010102682835167798e-02
            3.27883535441253481934609226300381e-02 4.17644719340453940414903399869218e-01
            2.18835813694288894293293878945406e-02 4.88963910362178622026618768359185e-01
            5.17741045072915850866479559044819e-02 2.73477528308838646609046918456443e-01
            ];
        
        S3=[];
        
        
    case  15
        
        % ALG. DEG.:   15
        % PTS CARD.:   49
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 4.8e-16
        
        S1=[
            2.22757447282223141477031091994832e-03 3.73744048757291905116240116058179e-02 9.62518352230012141923509716434637e-01
            2.70101416598694694937154991976058e-02 2.48248777984673207885890633406234e-01 1.93166698545214177151763124129502e-01
            2.60837796395875648469697694054048e-02 2.06994022748302169967260510929918e-01 8.68959088354996217473669162245642e-02
            1.21101532770282842299547354514289e-02 1.48541105269547085709902489725209e-01 1.74368253984543072898638627066248e-02
            1.56478505968044440122000082737941e-02 3.06742379235963813499665775452740e-01 1.74925109582576600308900793834255e-02
            3.41708893792947929202163948048110e-02 3.67031987542204729546568842124543e-01 9.03480217586455636169517902089865e-02
            ];
        
        S2=[
            1.51731495572117042192061830974126e-02 1.10222296228346866575620310868544e-01
            1.29760012839288414038252383875260e-02 5.19764330100343507901072825916344e-02
            1.70662959680061565581610238950816e-02 4.91145658075325564073665418618475e-01
            4.57600194627376041789368343870592e-02 3.93157188884358854696898788461112e-01
            ];
        
        S3=[
            2.35712670319063420654170926127335e-02
            ];
        
        
    case  16
        
        % ALG. DEG.:   16
        % PTS CARD.:   55
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 8.1e-16
        
        S1=[
            1.43003329044953646914661860023443e-02 6.50513402661352269085170973994536e-01 3.31399744537089535612039981060661e-01
            2.78497772036008298113252124039718e-02 6.04011281495997032209288590820506e-01 3.03247162749942178461282082935213e-01
            7.04167340663609721967963750444142e-03 8.02168257574741638826765210978920e-01 1.88028059521237161932916137629945e-01
            1.78998382599337285681073694831866e-02 7.56505606442828382185439295426477e-01 1.83504668522296854327180426480481e-01
            2.74582003843497624018255010014400e-02 4.65938438714118163819222218080540e-01 3.59645948797504610983111206223839e-01
            7.29979693943176230486180600109947e-03 9.06394843992041510105650559125934e-01 7.71943712957554317721786674155737e-02
            ];
        
        S2=[
            1.47091003068019268718025216458045e-02 8.17949831313738717764039165558643e-02
            2.95445865493192544315448344605102e-02 1.65300601969779636934987365748384e-01
            2.61250173510883774985469329976695e-02 4.68592105349461385266351953760022e-01
            2.78038735239000698098998931584447e-03 1.44388134454166826986520888453924e-02
            3.18217730005366516987663771942607e-02 2.41784285391783343444060960791830e-01
            8.64583434950966060739130369938721e-03 4.95310342987769991474067410308635e-01
            ];
        
        S3=[
            4.80221886803770883767050747792382e-02
            ];
        
        
    case  17
        
        % ALG. DEG.:   17
        % PTS CARD.:   60
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.2e-16
        
        S1=[
            3.08496545825140619512261253021279e-02 1.09865907082626154678273167064617e-01 3.04665769698665711562313163085491e-01
            1.88139854400541997603291122231894e-02 2.04932274629187893832948930139537e-01 5.24875839064542562550919058139698e-02
            5.12343450397285526259905097390401e-03 5.81392156426624381881929082283023e-02 1.50005399522595440214844941806405e-02
            7.01239348475201761806685496480895e-03 1.38595540867764821513219430926256e-01 1.50102334797318250070308209842551e-02
            1.53822944350446126626774656642738e-02 3.46605469520092579482906103294226e-01 2.33621289331465385730179917800342e-02
            3.03013148261713129855143833424336e-03 2.48219868895855927393157003280066e-01 9.99999999999657541679184274974190e-07
            ];
        
        S2=[
            3.82925400800356863229012560623232e-02 2.40563069636269027951058774306148e-01
            1.66952869977533974177319464615721e-02 8.09232358976607335510067287032143e-02
            1.43512454359061227167349006350605e-03 1.00141491249913505701130134184496e-02
            2.86427684918505350009088061824514e-02 1.54376520786632887016764925647294e-01
            3.40856907820621529703331020755286e-02 4.17169862019962700561137580734794e-01
            2.46727420005308922812403693569649e-02 4.70869745738400957701941251798416e-01
            5.86679757537134170114478948221404e-03 4.98118033845422025773785890123690e-01
            2.32185950042289625250280238333289e-02 3.64738405652919261967781494604424e-01
            ];
        
        S3=[];
        
        
    case  18
        
        % ALG. DEG.:   18
        % PTS CARD.:   73
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.0e-15
        
        S1=[
            3.88487690498138995426335995375666e-03 6.00633794794644959580409704358317e-01 3.95754787356942860920838711535907e-01
            2.55741606120219039377516878630558e-02 1.34466754530779775134163855909719e-01 5.57603261588783927216184110875474e-01
            8.88090357333805834050277638880289e-03 7.20987025817365068114384030195652e-01 2.64566948406520230374638913417584e-01
            1.61245467617313897878972284161136e-02 5.94527068955870929833906757266959e-01 3.58539352205950567764602965326048e-01
            2.49194181749067553491294724210547e-03 8.39331473680838602291487404727377e-01 1.57807405968594743894684029328346e-01
            1.82428401189505783042132947002756e-02 2.23861424097915678865788891016564e-01 7.01087978926173360783025145792635e-01
            1.02585637361985206050807306610295e-02 8.22931324069856584557669521018397e-01 1.42421601113383450298854882021260e-01
            3.79992885530191408863243118787523e-03 9.24344252620784012641763638384873e-01 6.54946280829376975551880946113670e-02
            ];
        
        S2=[
            1.03307318912720536713267804884708e-02 4.89609987073006347468862031746539e-01
            2.23872472630163923956736482523411e-02 4.54536892697892658254943398787873e-01
            3.02661258694680718184244483381917e-02 4.01416680649431201199917040867149e-01
            3.04909678021977811890330656297010e-02 2.55551654403097627810126368785859e-01
            2.41592127416409042928169270680883e-02 1.77077942152129547137562326497573e-01
            1.60508035868008741564594288320222e-02 1.10061053227951857746980124375114e-01
            8.08458026178406063966264127884642e-03 5.55286242518396727652074673642346e-02
            2.07936202748478090804318263451478e-03 1.26218637772286691040024564358646e-02
            ];
        
        S3=[
            3.29063313889186512262163830655481e-02
            ];
        
        
    case  20
        
        % ALG. DEG.:   20
        % PTS CARD.:   82
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 9.1e-16
        
        S1=[
            7.66503799888173468352059458652548e-03 4.23177101943935984884603840328055e-01 5.66246814212167381974438740144251e-01
            6.11096641377269558792484360765229e-03 2.87178034937351334132671354382182e-01 7.03631343750081739152335558173945e-01
            3.29750122890750362036893506001434e-03 1.62824865710703176047147167082585e-01 8.32373311160547357800965073693078e-01
            2.58504328745784222806647356662779e-03 6.46806344981751130118752257658343e-02 9.29257699458375552303834865597310e-01
            1.60482996634683378056607239159348e-02 3.34006277005119089906770568632055e-01 6.13237016399387990794878078304464e-01
            9.87379147789446687255576762254350e-03 2.10923795524181278970843322895234e-01 7.51510514906010507196754133474315e-01
            8.17314741769008083760450489307914e-03 1.15080380841368318245443447267462e-01 8.50151131562102868599595240084454e-01
            2.12721242081270779655799429974650e-02 3.10752086465084287336679835789255e-01 5.61989749539536176392573452176293e-01
            1.62955996807267848580647040535041e-02 2.02006198010458992708748837685562e-01 7.01716440236161864696384782291716e-01
            1.91054178147478798122094190148346e-02 2.89023207908959300915796575282002e-01 4.91388562323192112923919694367214e-01
            ];
        
        S2=[
            1.70118788706517913567051181189527e-02 4.72539690493749442445903241605265e-01
            2.21346290253984709361922966763814e-02 4.35591797658190560404989355447469e-01
            2.22501203414893662657636497215208e-02 3.85483177690954081917595885897754e-01
            2.10880142751876560602841692571019e-02 1.85897872669382596688691933195514e-01
            1.43667319823725068983488739604581e-02 1.02943093872275681643913003426860e-01
            5.96064247309054887763446473059048e-03 4.42043568221049953947954236355145e-02
            1.85446185638856223497350494966440e-03 1.18770000819499090688458053932663e-02
            ];
        
        S3=[
            2.34389883762168536873726765179526e-02
            ];
        
        
    case  21
        
        % ALG. DEG.:   21
        % PTS CARD.:   87
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 5.6e-16
        
        S1=[
            3.24321603629715857555759761510217e-03 1.01703755139702384929778489208729e-02 5.61060787404411287493388726943522e-02
            4.44606682456729577351817184194260e-03 7.40148001912942655328153662708246e-03 2.38060908939278198515054896233778e-01
            4.44713701276898583369945328058748e-03 9.48888021003903639472021325218520e-03 1.33714236127292523947929225869302e-01
            7.19283008979440269953231634758595e-03 1.07870852895329272236324413825059e-02 3.60878270361336406235608365022927e-01
            9.68818276967735834970252284392700e-03 4.88864101564119649112605259233533e-02 1.28560007074346699385003489624069e-01
            1.07718672547460047467238197782535e-02 4.07904784349213886396334771688998e-02 2.31929220198316188294995754404226e-01
            1.53764487922741950598481963652375e-02 1.02254765254272128593981960875681e-01 2.13182380246852126726153642266581e-01
            1.54182016250509965060722095131496e-02 5.61951037264934380455194684600428e-02 3.46153638930814133711777458302095e-01
            1.87279940818234902355499116310966e-02 1.93921114022049395630986623473291e-01 3.57417794125976018282386803548434e-01
            1.96189079783958622571482521834696e-02 1.31396937599549262776221780768537e-01 3.09641800485391083253006172526511e-01
            ];
        
        S2=[
            1.53252214157029924106179219478463e-03 1.08306479138961346542213703969537e-02
            2.95964208580089043962302319812352e-03 4.98681169280067615456886187530472e-01
            7.10507195198322675372404333415943e-03 5.32063025937022163192047230495518e-02
            1.24220254501511920969258895297571e-02 4.84368806799071960345060006147833e-01
            1.31538530905486034106033343960007e-02 1.17179324464095319102696635127359e-01
            1.61071199335594099355528641126512e-02 1.86199874531600445548562561270955e-01
            1.95569983274215299096088216401768e-02 2.37705046970502376080247586287442e-01
            2.11471458034111527146681197564249e-02 4.51118096863331430323995618891786e-01
            2.14872496180955330213890874802019e-02 2.98901616618379606382660540475626e-01
            ];
        
        S3=[];
        
        
    case  22
        
        % ALG. DEG.:   22
        % PTS CARD.:   96
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.7e-16
        
        S1=[
            1.82862802731120917786034052454625e-03 7.73958468860307807757337172915868e-03 3.91001678273283775899571423906309e-02
            1.88044970396283829192074499303544e-03 5.77064130218403839069074567902362e-04 2.99563010316006073097128137305845e-01
            3.70410864001110206808475844297845e-03 9.79577185352314104560544905098141e-03 9.91675808522524682153687081154203e-02
            4.66834037183480599075835471012397e-03 9.18868020649896491414132526642788e-03 1.86190681552805714638054723764071e-01
            7.21134117192958560421667257855916e-03 1.34670875684018177820266970456942e-02 4.02938891855100456140803544258233e-01
            7.96743335988734334429661743115503e-03 5.06263124673368980488419310859172e-02 1.03588085409291874583459502900951e-01
            9.03580036943713547914480699319029e-03 4.83881177618376498439367594528449e-02 1.80823006888328891017536648178066e-01
            9.48064111155901521887923166787004e-03 2.74914358532603458895593462329998e-02 2.81943415079491888075580163786071e-01
            1.56173701283532084804761552732089e-02 6.71339744477043864900878133994411e-02 3.50712853397991353165963346327771e-01
            1.62453191810569830266786084393971e-02 9.67043036168603459268311439700483e-02 2.30176404403601320147743081179215e-01
            2.17848082235315120003260602743467e-02 1.51930454826702282522532527764270e-01 3.11593235863486472947414540612954e-01
            ];
        
        S2=[
            6.97145696250740256293132457443562e-04 7.22823551590805277872187062371268e-03
            1.73365376198036241772726384624548e-03 4.99714916562231881158595570013858e-01
            4.66131304304557344403425744872038e-03 4.19525272728040882208588868707011e-02
            1.15845767655995638550914250686219e-02 3.14132191520500925197012520584394e-01
            1.18789740330219103364717270210349e-02 4.80013651617116565706311348549207e-01
            1.53388036556666995657849383860594e-02 1.23087454230161152946010361119988e-01
            1.97163769402723364210228140791514e-02 4.45719630439115332976030003919732e-01
            2.05721514917347539364023134567105e-02 1.89862412183893686234981146299106e-01
            2.38406602781370408061789589737600e-02 2.60339233381755796870038466295227e-01
            2.44611970898748719294157893955344e-02 3.96366847671625999183220301347319e-01
            ];
        
        S3=[];
        
        
    case  24
        
        % ALG. DEG.:   24
        % PTS CARD.:  112
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.3e-15
        
        S1=[
            1.74618458266172162270113421556061e-03 1.28915765322182144549112425835347e-03 3.89300321173704000354831578079029e-01
            1.83857160374278731651553897563645e-03 8.04615544134750990123983171997679e-03 3.99455189985108982675399147410644e-02
            1.99893663336212900477617537831065e-03 3.97463904079080419112779409829272e-03 1.77567553518447190930018564358761e-01
            2.06369617133590798133124799562665e-03 5.65339683877998013666887899830726e-03 9.73043731633908004496902321989182e-02
            2.29539508715841588867845324273276e-03 3.65416463470007168989894807964447e-02 4.73791364695558270270225875719916e-02
            4.57952328890823181367641936390100e-03 8.47203125278924405738845138102988e-03 2.76542334906226316970645484616398e-01
            5.29323009909540322104337661812679e-03 3.18687855584848661605334996238525e-02 1.01860048008413445197462010582967e-01
            5.57874699599088166163118884810501e-03 2.50744857569593221358061896353320e-02 1.79900411429897849613368521204393e-01
            8.61933948725664145218416933857952e-03 2.49546492072369834047673720078819e-02 3.74761488291451505538987021282082e-01
            9.99235399224706992427069707218834e-03 7.18069274899266751788218243746087e-02 1.62853713656418902511191504345334e-01
            1.03056162076813079969683784042900e-02 4.64953683252256055880025087390095e-02 2.62999946925326910651676826091716e-01
            1.46045461262502476035285425837174e-02 7.94456583289277762371938251817483e-02 3.53874622686032425100677301088581e-01
            1.49985006762504632926713554752496e-02 1.12104968398914525695708732655476e-01 2.40452720011096854335619354969822e-01
            1.92307831174366797299946085786360e-02 1.62887044320893126236882153534680e-01 3.16328836361366161344932379506645e-01
            ];
        
        S2=[
            7.67377461115814228295728405271348e-04 7.64481587209140354965963837230447e-03
            5.52357844475169761783162059032293e-03 4.95123266934894168400660419138148e-01
            8.38499530275631059506036280026819e-03 8.53357147077405536750305259374727e-02
            1.22173801780023523477769487044498e-02 4.75404357579430003610809762903955e-01
            1.30703906056117823836215663391158e-02 1.43613126002549662540985764280776e-01
            1.78812277584295790944235449160260e-02 4.40001375500573721488706269155955e-01
            1.82217441390973081127846455729014e-02 2.04386847871303928281250250620360e-01
            2.16414972175676158239188140441911e-02 3.91555207032681984458122315118089e-01
            2.16883129269226024304906275119720e-02 2.69360691849568234523815135617042e-01
            ];
        
        S3=[
            2.29379434809674903616993901778187e-02
            ];
        
        
    case  25
        
        % ALG. DEG.:   25
        % PTS CARD.:  126
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 6.2e-16
        
        S1=[
            1.24828919927740412260108726627550e-03 1.29235270444219584907996623002191e-03 2.27214452153364110387201435514726e-01
            3.40475290880302636220555712043279e-03 5.39970127211620177370976492170485e-03 4.35010554853571707045745142750093e-01
            3.35965432606405477525268743477227e-03 6.38400303397498956187172680643016e-03 3.20309599272204481579251478251535e-01
            1.71615653949675774601890676507310e-03 5.02821150199308283118559970148453e-03 9.17503222800053058749725209963799e-02
            1.48085631671560275825216823619712e-03 6.82675862178184610778997054580941e-03 3.80108358587244316018072254337312e-02
            3.51131261072868024186721669366307e-03 1.00161996399295120940831083089506e-02 1.57425218485311796401404649259348e-01
            7.39355014970648899419458643933467e-03 2.57578131733900463073982933792649e-02 2.39889659778533270717559844342759e-01
            7.98308747737656691778607154219571e-03 3.02278981199158233716950405778334e-02 3.61943118126060559003320804549730e-01
            4.35596261315803791325240723608658e-03 3.05049901071620770121040067124341e-02 8.35519609548284608591117716969165e-02
            7.36505670141783096854792844965232e-03 4.59565473625693221570998048264300e-02 1.48443220732418112994821512984345e-01
            1.09635728464195451176133033754923e-02 6.74428005402776115095875297811290e-02 2.83739708727534956089755269204034e-01
            1.17499617435411225080388319952363e-02 7.00450914159106152734324268749333e-02 4.06899375118787598459846321929945e-01
            1.00156007137985770399923879381276e-02 8.39115246401166414713657104584854e-02 1.94113987024892492572192281841126e-01
            1.33096407876286682014921325389878e-02 1.20375535677152700753111957965302e-01 3.24134347000703204599858509027399e-01
            1.41544465052261311116543751609242e-02 1.48066899157366704820049108093372e-01 2.29277483555981043128468854774837e-01
            1.48813795611680028257994834461897e-02 1.91771865867325114196617619199969e-01 3.25618122595983827682886158072506e-01
            ];
        
        S2=[
            8.00558188002042231401489402742300e-03 4.86026758463412866806407919284538e-01
            1.59470768323904908692068005393594e-02 4.34410699336174210838379394772346e-01
            1.31091412307955037380713392280995e-02 3.89889135243963846910730808303924e-01
            1.95830009656356059111192990940253e-02 2.98443234019804493062366645972361e-01
            1.64708854415372626944780165558768e-02 2.34044172337371847447684558574110e-01
            8.54727907409214178457990129800237e-03 1.51468334609017596381619341627811e-01
            8.16188585722650052078908089470133e-03 1.12733893545993488216616640329448e-01
            6.12114653998375047444513441519121e-03 7.77156920915262061644313007491292e-02
            2.90849826493666053020548112328925e-03 3.48930936142969050361983818220324e-02
            6.92275245661998781324109053514348e-04 7.25818462093238882942847212120796e-03
            ];
        
        S3=[];
        
        
    case  26
        
        % ALG. DEG.:   26
        % PTS CARD.:  133
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 8.5e-16
        
        S1=[
            3.69419331787613960255134415433531e-03 4.39945211557685067482736940291943e-01 5.53674216445258049468236549728317e-01
            3.43799094677204935752157233253001e-03 3.29997034833231284167709418397862e-01 6.63643196738745944074366889253724e-01
            2.86153444220796880637092129973098e-03 2.30829591569331249445440334966406e-01 7.63088635213633348364226094417972e-01
            2.03926220043675488630130487877068e-03 1.47944114085407502079405617223529e-01 8.46742931842361046257394718850264e-01
            2.12260415935469279324898650429532e-03 8.33105116593902267352333979033574e-02 9.09277521720756043421829417638946e-01
            1.19562615477588653212226432742682e-03 3.55026032675821653916692355323903e-02 9.58595570159293530743127575988183e-01
            6.85739162489405404410369726520003e-03 4.37992850963974222899821597820846e-01 5.28750100020195179517656924872426e-01
            7.34359958211473762279553056941950e-03 3.41845115301239788063725200117915e-01 6.25010662557993978971637716313126e-01
            6.92380910755335723244519030572519e-03 2.43219913165517859221509411327133e-01 7.24644701363325371978874045453267e-01
            5.68034562903817728807265297064077e-03 1.53693911780638098329987428769527e-01 8.16687808118266356238734715589089e-01
            4.82902212700270921930112422160164e-03 8.18282484051558456350505821319530e-02 8.81377376506472409722903194051469e-01
            1.21701611161561095569849655362304e-02 4.05435287023867385602926560750348e-01 5.14442885044733677624151368945604e-01
            1.09339847002783221996402573950036e-02 3.00475234451001216307020058593480e-01 6.20596936642523910876434456440620e-01
            8.74472678661106748998488313873167e-03 2.10298774983056657728042182498029e-01 7.12604684013138367504325287882239e-01
            6.80604517250698801134056736827915e-03 1.41929497236802870663296971542877e-01 7.83644226369100249662835722119780e-01
            1.41159091974328836788243890509875e-02 3.22825706055321792753431964229094e-01 5.35389696333471332145848009531619e-01
            1.31147856257194574819413546151736e-02 2.25038500867665025584329896446434e-01 6.33993341119415743101228599698516e-01
            1.29403394597649903235980772819858e-02 3.06228014522925306994949323780020e-01 4.80239008804562494958645402221009e-01
            ];
        
        S2=[
            1.56284773881481507673552044934695e-02 4.27380776384648330967763740773080e-01
            1.46375262291117830271147681742150e-02 3.86991728384621147540656238561496e-01
            1.19987734663434490040012647682488e-02 2.90801606496775999488590969122015e-01
            1.40892750357101397690184896305254e-02 2.19551193915611925211450738970598e-01
            1.13708950654174238692473153378160e-02 1.41477668767692438267502552662336e-01
            5.76148405319945079716692504234743e-03 8.52977390247050409888274202785396e-02
            2.88047484993807917375008287308447e-03 3.24537987119420390236257389915409e-02
            6.17924007806828590010084223393960e-04 6.88528565235726423138240548382782e-03
            ];
        
        S3=[
            1.41775216099960243987254671083065e-02
            ];
        
        
    case  27
        
        % ALG. DEG.:   27
        % PTS CARD.:  145
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.0e-15
        
        S1=[
            2.98280618704304181787456684560311e-03 4.44334487696719404592471391879371e-01 5.50169162481872864312038018397288e-01
            2.86890029157565789805217626451395e-03 2.42719887748324536502764203760307e-01 7.51198680548561092962245311355218e-01
            1.46771786779737400084611120831823e-03 1.58188924752096032877446418751788e-01 8.38350695312842342232784176303539e-01
            1.63545643437809683565231644308824e-03 8.84556064909017975006477740862465e-02 9.06294109675106485113360577088315e-01
            3.26270663499477744581844795845882e-03 3.39993168915069976510068272546050e-01 6.53721870992564402236268961132737e-01
            1.27830372794885825833699488640605e-03 3.67115484722965942920858140041673e-02 9.57246043701164284200899601273704e-01
            3.55054931897130689882891019237832e-03 8.12402961036135656813783612051338e-02 8.91779193842819672966015787096694e-01
            7.05687847901615480233905586260335e-03 4.31949991404557809282493963110028e-01 5.38923331029669272496107623737771e-01
            7.10874754932738772694733597745653e-03 3.26996931617198338670959856244735e-01 6.40470902013947407382943310949486e-01
            6.19331568447356855394714258977729e-03 2.34081170727359955119339929296984e-01 7.34315971485084095782269741903292e-01
            4.30573263634848567293911258957451e-03 1.53473632723084674189806264621438e-01 8.24922676183995506882240533741424e-01
            5.38152713709093722532772474664853e-03 1.26999415890300532128165400536091e-01 8.16884834249502755909588813665323e-01
            9.46147607081304457821158138131068e-03 3.68066602265455233844448912350344e-01 5.59340535857428933219637201545993e-01
            1.03877644118448644872110264714138e-02 2.69796071148512450754708424938144e-01 6.50885906365806787832184454600792e-01
            6.93741437082199055574882606833853e-03 1.86994710857735058073814116141875e-01 7.44996011981384320677079813322052e-01
            1.05678608749397145299964151377026e-02 3.45950442872727914611630239960505e-01 5.27094080400327680280270215007477e-01
            1.00159152041039579278391968841788e-02 1.82456322380054797305248825978197e-01 6.93857687409184187288246903335676e-01
            1.14577693947056618445357045743549e-02 2.60183849659498422912662363160052e-01 5.94783798861356349796380982297705e-01
            1.29770901446694120195957111718599e-02 3.19848673583135789666442860834650e-01 4.82373460310693780161983568177675e-01
            ];
        
        S2=[
            1.01760677725488048878110802775154e-02 4.64270848054669227344959381298395e-01
            1.35536351895579262882840509973903e-02 4.32298314340281597534243474001414e-01
            1.43557523235289877239218014892685e-02 3.90753814771931773375257535008132e-01
            1.76215402547928445819991338794352e-02 2.99849030838122532216516447078902e-01
            1.28645661240754053539214751822328e-02 2.42386093366695004602817675731785e-01
            1.02112078872144199093163763336634e-02 1.95751633844707950693475595471682e-01
            8.05484559136441374826986816515273e-03 1.10964220718018535816185021758429e-01
            4.85083247712445042587603438732913e-03 6.78494378207233628241823453208781e-02
            2.76034923845063645542197150462016e-03 3.24881443144369105868740632558911e-02
            6.46520224024233554523111155987181e-04 7.02227048619495294867753898415685e-03
            ];
        
        S3=[
            1.32645422676788259791491597638924e-03
            ];
        
        
    case  28
        
        % ALG. DEG.:   28
        % PTS CARD.:  154
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.1e-15
        
        S1=[
            2.06901772064515534696460541397300e-03 2.95361534971414929628963363938965e-01 7.00853506852401775262251248932444e-01
            2.15432386268054427633233949279656e-03 2.02876702155597610222059756779345e-01 7.91935619745775754019234682345996e-01
            1.66065394763716990544910601101947e-03 1.28179495236313306660846933482389e-01 8.67015046983888204579216107958928e-01
            1.38084032913645066500962776956385e-03 7.05254410920549829056014345951553e-02 9.23919864914773625308441751258215e-01
            3.60171025661803296744056801514944e-03 3.95326317408967009559006555718952e-01 5.97713061106773357877841590379830e-01
            9.73861140945066184292089328522479e-04 2.95682731691414360519765125445701e-02 9.64652531132313861839122637320543e-01
            5.35413394306626013591232648991536e-03 3.06750949986907162081450906043756e-01 6.68500829530133566791505472792778e-01
            4.23736286661477445925072515819920e-03 2.25171823133371223901733060301922e-01 7.51322633820259988546297336142743e-01
            3.30847570421703100068677727563227e-03 7.63589242705955267842554690105317e-02 8.96119411373265895015549631352769e-01
            4.48583942101861074536195417294948e-03 1.40424755319720706570052470851806e-01 8.33282278597101022654669577605091e-01
            7.76555845120818939719686824219025e-03 3.91187470528715885986770217641606e-01 5.70828988946343374166758621868212e-01
            6.87043734971177089826399608796237e-03 1.93960811129872207070334866330086e-01 7.50102714981103257940731054986827e-01
            5.84068837096601671532436483857964e-03 1.15860207213215674060258209010499e-01 8.17774906244078891859317081980407e-01
            8.88657317183860415155116641017230e-03 2.80237285745932929170720626643742e-01 6.55282643699637845635663779830793e-01
            1.03831577856246647789673076545114e-02 3.54893347574873008198892421205528e-01 5.57875630565633384705392927571665e-01
            6.55128753862388706757924339285637e-03 1.45342107354678340414011472603306e-01 7.49654646588252315808631465188228e-01
            8.77076875255828995736617770262455e-03 2.13535944606159017178370618239569e-01 6.78156200353984717210664712183643e-01
            1.21889293572907293772278336518866e-02 3.89059867579780116475518525476218e-01 4.74786102188072933749651838297723e-01
            1.25064354962984430885697406665713e-02 2.75628902229600292184130694295163e-01 5.86030082000936203279195524373790e-01
            1.26984755922827027840860125706968e-02 3.13779953299047120740539185135276e-01 4.91511463511942858772840736492071e-01
            ];
        
        S2=[
            1.93654415522499095958508519288443e-03 4.98353413126779720698777964571491e-01
            6.05141039266348344793922109374762e-03 4.88562754090508810644877257800545e-01
            1.14141135507166389950750584603156e-02 4.64781478287247362679579509858740e-01
            1.60800737631301288166341123542225e-02 3.89766273321017708841651483453461e-01
            1.40353291043240711172268575523958e-02 3.01546751182982020189626837236574e-01
            9.78327794286426230541131587870041e-03 2.61574074059258976365782700668206e-01
            1.26783016982056004340329025126266e-02 2.14107886345909520509067647253687e-01
            1.01141158594697452843824692081398e-02 1.70138191271475530674806009301392e-01
            3.76318422499545210402338568655978e-03 6.43707835026944286127559280430432e-02
            2.43119330615522129604433310134937e-03 3.05600321040132086314322634734708e-02
            4.25889945316464254773669573239658e-04 5.70940793308473588940632481580906e-03
            ];
        
        S3=[
            3.72851181690746821123516596685477e-03
            ];
        
        
    case  29
        
        % ALG. DEG.:   29
        % PTS CARD.:  166
        % NEG. W.  :    0
        % OUT PTS. :    0
        % M.E.INF. : 1.3e-15
        
        S1=[
            1.29189419129147292468473562365716e-03 3.24591680276287153361636228510179e-01 6.73712121758946347327423609385733e-01
            1.12772642035394166093453804933233e-03 2.05382961103950267833795351180015e-01 7.92855850980322518495313488529064e-01
            1.84745171735374840576993094032332e-03 1.27817155048788905213896782697702e-01 8.66822611309806201340677489497466e-01
            7.17822033246549406730607767457286e-04 6.39721317044320281208769074510201e-02 9.34220030168086745803179837821517e-01
            3.51318664214703190243338504217263e-03 4.08772703046101637536224870927981e-01 5.83304257472950338225814448378514e-01
            3.49660600844854327351463041395618e-03 2.60003660015059478993748598441016e-01 7.28202443651907782573573513218435e-01
            2.41690948038281259308002191232845e-03 8.37010376047120779885446495427459e-02 8.98464666406439738111089354788419e-01
            8.13134666139966075064371420211273e-04 2.27772424394371494904554253935203e-02 9.71586291829570680711469776724698e-01
            5.64942974660398271324890373534799e-03 3.32497033049334811405373102388694e-01 6.42488243894240529385797344730236e-01
            4.45904031659885048688440889463891e-03 1.75467715977808663696535518283781e-01 8.03433175906504426500021054380341e-01
            1.85640152529500013396657198683215e-03 4.23914433713915364543289854282193e-02 9.37282237421026453993988525326131e-01
            7.01742231395962624829198972520317e-03 2.38086960749264359149535152937460e-01 7.16981908850933602472821348783327e-01
            4.76963067992196287914197228019475e-03 1.13850259493195438142976172457566e-01 8.44311313344634828403911797067849e-01
            7.68181583252764647179455792525005e-03 4.07699471210733355164279601012822e-01 5.48247215417901112566312349372311e-01
            8.66304121915658330299603306912104e-03 3.09719966010297276248763864714419e-01 6.22325578952138247501579826348461e-01
            6.94192073189620235951213800262849e-03 1.65012577073424038243132372372202e-01 7.67767929108697733120436623721616e-01
            6.08713825993680707204802615706285e-03 4.37364893469387783220270193851320e-01 4.84688748698669014114415176663897e-01
            9.87867211691082283686604625927430e-03 2.30361692319617394897335316272802e-01 6.66275923025302252433732519421028e-01
            1.04031774001664949896861500633349e-02 3.53054767022387583352838191785850e-01 5.40538992741738999470157978066709e-01
            6.77295750139824629526552612901469e-03 1.58718755804575895052010992003488e-01 7.21198858207568349598659551702440e-01
            8.32356466794721694402170442117495e-03 4.02149591942053019621994280896615e-01 4.54827385300507591736618451250251e-01
            1.17582567855695832581552906503930e-02 2.71868486061960623612066001442145e-01 5.77168860612755296735087995330105e-01
            1.26811781993145834823710771388505e-02 3.28703899955314460257227437978145e-01 4.78814632497088499274440209774184e-01
            ];
        
        S2=[
            1.19407074555933690775122446581236e-03 4.99416262867999560803866643254878e-01
            5.58417807814421719714292891012519e-03 4.89483469832614981331886383486562e-01
            1.40142965005035178421843866658492e-02 3.83374906469260035013491005884134e-01
            1.41078060470803436088349513966023e-02 2.86579671246249501059821795934113e-01
            1.34281043157659131515035966231153e-02 2.36757437980759072315706248446077e-01
            1.21753749760933841633780971847045e-02 1.84448542625493788760238089707855e-01
            7.22294427143959326398370279775918e-03 9.59209723834500288530691136656969e-02
            4.43751867960735617291145871377012e-03 5.73266371488608539319642432019464e-02
            1.97718815192656101591833772701534e-04 3.60814709934401383478741465182793e-03
            ];
        
        S3=[
            1.39036919724349807697683090168539e-02
            ];
        
        
        
    otherwise
        S1=[]; S2=[]; S3=[];
end




pointset_stats=[
    1    1 1.92e-16    0    0
    2    3 3.68e-16    0    0
    3    6 1.46e-16    0    0
    4    6 1.39e-16    0    0
    5    7 1.98e-16    0    0
    6   12 2.36e-16    0    0
    7   16 3.75e-16    0    0
    8   21 4.42e-13    0    0
    9   19 5.55e-16    0    0
    10   25 4.72e-16    0    0
    11   28 8.33e-16    0    0
    12   33 4.82e-16    0    0
    13   37 5.83e-16    0    0
    14   42 4.58e-16    0    0
    15   49 4.82e-16    0    0
    16   55 8.12e-16    0    0
    17   60 6.21e-16    0    0
    18   73 1.01e-15    0    0
    20   82 9.06e-16    0    0
    21   87 5.59e-16    0    0
    22   96 6.66e-16    0    0
    24  112 1.34e-15    0    0
    25  126 6.17e-16    0    0
    26  133 8.50e-16    0    0
    27  145 9.99e-16    0    0
    28  154 1.11e-15    0    0
    29  166 1.33e-15    0    0
    ];




