
function demo_AFP_DLP_disk

set_type=1; % 0 AFP, 1 DLP.
degV=1:20;

X0=wam_disk(100);

diary on;

for ii=1:length(degV)
    
    deg=degV(ii);
    deg_wam=max(100);
    
    XW=wam_disk(deg_wam);
    
    X=points_extraction(XW,deg,set_type);
    VX=vandermonde_koornwinder_typeIIbis(X,deg);
    VX0=vandermonde_koornwinder_typeIIbis(X0,deg);
    leb_const=determine_lebesgue_constant(VX,VX0);
    
    printstats(X,deg,leb_const,VX);
    
end

diary off;






function pts=points_extraction(XW,deg,set_type)

% XW: INITIAL MESH: wam.
% set_type: 0 AFP, 1: DLP.

omega=ones(length(XW(:,1)),1); V_XW=vandermonde_koornwinder_typeIIbis(XW,deg);
%[V_XW,omega]=compute_vandermonde(deg,XW);

% polynomial basis orthogonalization
[Q,R]=qr(V_XW,0);
% tiny imaginary parts may appear
Q=real(Q);
% possible re-orthogonalization
% [Q,R1]=qr(Q,0);
% moments of the orthogonal basis
orthmom=Q'*omega;

switch set_type
    case 0
        weights=mldivide(Q',orthmom);
        ind=find(abs(weights)>0);
    case 1
        dim=(deg+1)*(deg+2)/2;
        [L,U,perm]=lu(Q,'vector');
        ind=perm(1:dim);
end


% indexes of nonvanishing weights and compression
pts=XW(ind,:);









function printstats(X,deg,leb_const,VX)

loc_type=2;

switch loc_type
    case 1
        fprintf('\n \t %3g  %5.3f',deg,leb_const);
    case 2
        fprintf('\n \n \t case %2.0f', deg);
        fprintf('\n \n \t %% DEG: %2.0f', deg);
        fprintf('\n \t %% LEB. CONST.: %1.3e', leb_const);
        fprintf('\n \t %% ABS. DET. VAND.: %1.3e', det(VX));
        
        XX=X(:,1); YY=X(:,2);
        norms=sqrt(XX.^2+YY.^2);
        iout=find(norms > 1);
        
        fprintf('\n \t %% OUT: %4.0f', length(iout));
        
        fprintf('\n \n \t pts=[');
        for ii=1:size(X,1)
            fprintf('\n \t %1.20e %1.20e',X(ii,1),X(ii,2));
        end
        fprintf('\n \t ]; \n');
end













function V=vandermonde_koornwinder_typeIIbis(pts,deg,a)

%--------------------------------------------------------------------------
% OBJECT.
%----------
%
% KOORNWINDER TYPE II BASIS ON THE UNIT DISK "B([0,0],1)".
%
%--------------------------------------------------------------------------
% INPUT.
%----------
%
% deg: VANDERMONDE MATRIX UP TO DEGREE d INCLUDED.
% pts: SETS OF POINTS OF THE UNIT DISK "B([0,0],1)".
% a:   KOORNWINDER EXPONENT (DEFAULT "a=-0.5").
%--------------------------------------------------------------------------
% OUTPUT.
%----------
%
% V: VANDERMONDE MATRIX w.r.t. THE KOORNWINDER TYPE II BASIS ON THE SET pts.
%
%--------------------------------------------------------------------------
% FUNCTIONS USED.
%----------------
%
% 1. r_jacobi
% 2. eval_mod_op
% 3. eval_op
%
%--------------------------------------------------------------------------
%%
%% Copyright (C) 2010. Alvise Sommariva, Marco Vianello.
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Author:  Alvise Sommariva <alvise@euler.math.unipd.it>
%%          Marco Vianello   <marcov@euler.math.unipd.it>
%%
%% Date: October 25, 2011
%%
%--------------------------------------------------------------------------

if nargin < 3
    a=0;
end

mod_op_type=3; % ORTHONORMAL JACOBI.
op_type=3; % ORTHONORMAL JACOBI.

x=pts(:,1); y=pts(:,2);

c=sqrt(1-x.^2);
t=y;

a1=a; b1=a;

ab=r_jacobi(deg,a1,b1);
V1=eval_mod_op(deg,ab,a1,b1,t,c,mod_op_type);

t=x; V=[];

for k=0:deg
    
    a2=a+k+0.5; b2=a+k+0.5;
    
    if deg-k > 0
        ab2=r_jacobi(deg-k,a2,b2);
    else
        ab2=[];
    end
    
    V2loc=eval_op(deg-k,ab2,t,a2,b2,op_type);
    V1loc=repmat(V1(:,k+1),1,size(V2loc,2));
    
    V=[V V1loc.*V2loc];
    
end





% 1. eval_op (line 100)
% 2. c_orthnrm (line 180)
% 3. c_standard (line 210)
% 4. eval_mod_op (line 240)
% 5. c_dub (line 320)


function V=eval_op(deg,ab,x,a,b,type)

%--------------------------------------------------------------------------
% OBJECT.
%----------
%
%
%
%--------------------------------------------------------------------------
% FUNCTIONS USED.
%----------------
%
%
%
%--------------------------------------------------------------------------
%%
%% Copyright (C) 2010. Alvise Sommariva, Marco Vianello.
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Author:  Alvise Sommariva <alvise@euler.math.unipd.it>
%%          Marco Vianello   <marcov@euler.math.unipd.it>
%%
%% Date: October 18, 2010
%%
%--------------------------------------------------------------------------

if nargin < 3
    a=0; b=0;
end

L=size(x,1);

if deg > 0
    aa=ab(:,1);
    bb=ab(:,2);
else
    ab=r_jacobi(1,a,b);
    aa=[];
    bb=[];
end

switch type
    case 1 % MONIC JACOBI POLYNOMIALS.
        V=[zeros(L,1) ones(L,1)];
        cn=ones(size(ab,1),1);
        co=ones(size(ab,1),1);
        
    case 2 % CLASSICAL JACOBI (SEE WOLFRAM HOMESITE)
        V=[zeros(L,1) ones(L,1)];
        [cn,co]=c_standard(deg,a,b);
        
    case 3 % ORTHONORMAL JACOBI POLYNOMIALS.
        V=[zeros(L,1) (1/sqrt(ab(1,2)))*ones(L,1)]; % "ab(1,2)" IS "int_-1^1 w(x) dx".
        [cn,co]=c_orthnrm(deg,a,b);
end

for ii=1:deg
    Vloc=cn(ii)*(x-aa(ii)).*V(:,ii+1)-co(ii)*bb(ii)*V(:,ii);
    V=[V Vloc];
end

V=V(:,2:end);





function [cn,co]=c_orthnrm(ade,a,b)

[cn_st,co_st]=c_standard(ade,a,b);

ab=r_jacobi(1,a,b);
an0=ab(1,2);

n=(1:ade)';

if length(n) > 0
    if abs(b) > 0
        anloc=(2^(a+b+1)).*gamma(n+a+1).*gamma(n+b+1)./((2*n+a+b+1).*gamma(n+a+b+1).*gamma(n+1));
    else
        anloc=2^(a+b+1)./(2*n+a+b+1);
    end
else
    anloc=[];
end

an=[an0; anloc];

an=an.^(-1/2);
an_rat=an(2:end,:)./an(1:end-1,:);
cn=cn_st.*an_rat;
co=[1; cn(2:end,:).*cn(1:end-1,:)];





function [cn,co]=c_standard(deg,a,b)

s=a+b;

% INITIALIZATION TO AVOID SOME POSSIBLE OVERFLOW PROBLEMS FOR NEGATIVE s.
if (deg > 0)
    cn1=0.5*gamma(2+s+1)/gamma(s+2);
else
    cn1=[];
end

n=2:deg; n=n';

if (abs(s-floor(s)) > 0)
    cnloc=(1./(2*n)).*(gamma(2*n+s+1)./gamma(2*n+s-1)).*(gamma(n+s)./gamma(n+s+1));
else
    cnloc=(1./(2*n)).*(2*n+s).*(2*n+s-1)./(n+s);
end

cn=[cn1; cnloc];

co=[1; cn(2:end,:).*cn(1:end-1,:)];







function V=eval_mod_op(deg,ab,a,b,t,c,type)

%--------------------------------------------------------------------------
% OBJECT.
%----------
%
%
%--------------------------------------------------------------------------
% FUNCTIONS USED.
%----------------
%
%--------------------------------------------------------------------------
%%
%% Copyright (C) 2010. Alvise Sommariva, Marco Vianello.
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Author:  Alvise Sommariva <alvise@euler.math.unipd.it>
%%          Marco Vianello   <marcov@euler.math.unipd.it>
%%
%% Date: October 18, 2010
%%
%--------------------------------------------------------------------------

if nargin < 3
    a=0; b=0;
end

L=size(t,1);
aa=ab(:,1);
bb=ab(:,2);

switch type
    case 1 % MONIC JACOBI POLYNOMIALS.
        V=[zeros(L,1) ones(L,1)];
        cn=ones(size(ab,1),1);
        co=ones(size(ab,1),1);
    case 2 % CLASSICAL JACOBI (SEE WOLFRAM HOMESITE)
        V=[zeros(L,1) ones(L,1)];
        [cn,co]=c_standard(deg,a,b);
        
    case 3 % ORTHONORMAL JACOBI POLYNOMIALS.
        V=[zeros(L,1) (1/sqrt(ab(1,2)))*ones(L,1)]; % "ab(1,2)" IS "int_-1^1 w(x) dx".
        [cn,co]=c_orthnrm(deg,a,b);
        
    case 4 % ORTHONORMAL JACOBI POLYNOMIALS USED BY DUBINER ORTHONORMAL BASIS OVER THE SYMPLEX.
        V=[zeros(L,1) (sqrt(8)/sqrt(ab(1,2)))*ones(L,1)]; % "ab(1,2)" IS "int_-1^1 w(x) dx".
        [cn,co]=c_dub(deg,a,b);
end

for ii=1:deg
    Vloc=cn(ii)*(t-c.*aa(ii)).*V(:,ii+1)-(c.^2)*co(ii)*bb(ii).*V(:,ii);
    V=[V Vloc];
end

V=V(:,2:end);











function [cn,co]=c_dub(ade,a,b)

[cn_orth,co_orth]=c_orthnrm(ade,a,b);
cn=2*cn_orth;
co=[1; cn(2:end,:).*cn(1:end-1,:)];








function xw=gauss(N,ab)
N0=size(ab,1); if N0<N, error('input array ab too short'), end
J=zeros(N);
for n=1:N, J(n,n)=ab(n,1); end
for n=2:N
    J(n,n-1)=sqrt(ab(n,2));
    J(n-1,n)=J(n,n-1);
end
[V,D]=eig(J);
[D,I]=sort(diag(D));
V=V(:,I);
xw=[D ab(1,2)*V(1,:)'.^2];






% R_JACOBI Recurrence coefficients for monic Jacobi polynomials.
%
%    ab=R_JACOBI(n,a,b) generates the first n recurrence
%    coefficients for monic Jacobi polynomials with parameters
%    a and b. These are orthogonal on [-1,1] relative to the
%    weight function w(t)=(1-t)^a(1+t)^b. The n alpha-coefficients
%    are stored in the first column, the n beta-coefficients in
%    the second column, of the nx2 array ab. The call ab=
%    R_JACOBI(n,a) is the same as ab=R_JACOBI(n,a,a) and
%    ab=R_JACOBI(n) the same as ab=R_JACOBI(n,0,0).
%
%    Supplied by Dirk Laurie, 6-22-1998; edited by Walter
%    Gautschi, 4-4-2002.
%
function ab=r_jacobi(N,a,b)
if nargin<2, a=0; end;  if nargin<3, b=a; end
if((N<=0)|(a<=-1)|(b<=-1)) error('parameter(s) out of range'), end
nu=(b-a)/(a+b+2);
mu=2^(a+b+1)*gamma(a+1)*gamma(b+1)/gamma(a+b+2);
if N==1, ab=[nu mu]; return, end
N=N-1; n=1:N; nab=2*n+a+b;
A=[nu (b^2-a^2)*ones(1,N)./(nab.*(nab+2))];
n=2:N; nab=nab(n);
B1=4*(a+1)*(b+1)/((a+b+2)^2*(a+b+3));
B=4*(n+a).*(n+b).*n.*(n+a+b)./((nab.^2).*(nab+1).*(nab-1));
ab=[A' [mu; B1; B']];










function pts=wam_disk(n)

%--------------------------------------------------------------------------
% OBJECT.
%----------
%
% COMPUTE WAM POINTS FOR UNIT DISK "B([0,0],1)".
%
%--------------------------------------------------------------------------
% INPUT.
%----------
%
% n:  WAM DEGREE.
%
%--------------------------------------------------------------------------
% OUTPUT.
%----------
%
% pts: WAM POINTS ON THE UNIT DISK "B([0,0],1)".
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
%% Copyright (C) 2010
%% Len Bos, Stefano De Marchi, Alvise Sommariva, Marco Vianello.
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Author:
%%          Len Bos           <leonardpeter.bos@univr.it>
%%          Stefano De Marchi <demarchi@math.unipd.it>
%%          Alvise Sommariva  <alvise@math.unipd.it>
%%          Marco Vianello    <marcov@math.unipd.it>
%%
%% Date: October 18, 2010.
%--------------------------------------------------------------------------

if (mod(n,2) == 0)
    % SOME ZEROS COULD BE REPEATED IN THE ORIGINAL FORMULATION THAT
    % HAS BEEN CONSEQUENTLY MODIFIED.
    j=0:n; j=setdiff(j,n/2); % WE ADD [0 0] LATTER.
    k=0:n;
    [rho,theta]=meshgrid(cos(j*pi/n),k*pi/(n+1));
    meshA=[rho(:).*cos(theta(:)) rho(:).*sin(theta(:))];
    % ADDING [0 0]
    nodes_x=[meshA(:,1); 0];
    nodes_y=[meshA(:,2); 0];
else
    % ALL POINTS ARE DISTINCT.
    j=0:n;
    k=0:n;
    [rho,theta]=meshgrid(cos(j*pi/n),k*pi/(n+1));
    meshA=[rho(:).*cos(theta(:)) rho(:).*sin(theta(:))];
    nodes_x=meshA(:,1);
    nodes_y=meshA(:,2);
end

pts=[nodes_x nodes_y];





function leb_const=determine_lebesgue_constant(V_pts,V_control_points)

%--------------------------------------------------------------------------
% OBJECT.
%--------------------------------------------------------------------------
%
%--------------------------------------------------------------------------
% INPUT.
%--------------------------------------------------------------------------
% V_pts: Vandermonde matrix at the pointset of which we are interested to
%        compute the Lebesgue constant.
% V_control_points: Vandermonde matrix at pointset in which the Lebesgue
%        function, relative to the poinset "pts", is evaluated.
%--------------------------------------------------------------------------
% OUTPUT.
%--------------------------------------------------------------------------
%
%--------------------------------------------------------------------------
% EXAMPLE.
%--------------------------------------------------------------------------
%
%--------------------------------------------------------------------------
% COPYRIGHT.
%--------------------------------------------------------------------------
%
%--------------------------------------------------------------------------

leb_const=norm(V_pts'\V_control_points',1);






function [V,w]=compute_vandermonde(deg,X,w)

if nargin < 3
    w=ones(length(X(:,1)),1);
end

rect(1)=min(X(:,1));
rect(2)=max(X(:,1));

rect(3)=min(X(:,2));
rect(4)=max(X(:,2));

if length(X(1,:)) == 3
    rect(5)=min(X(:,3));
    rect(6)=max(X(:,3));
end;


V=chebvand(deg,X,rect);











function V = chebvand(deg,gmesh,rect);

% computes by recurrence the Chebyshev-Vandermonde matrix on a 2d or 3d
% arbitrarily located mesh, in the total-degree product Chebyshev basis
% of a given rectangle with the graded lexicographical order

% October 2015


% INPUT:

% deg = polynomial degree

% gmesh = 2- or 3-column array of mesh point coordinates

% rect = 4- or 6-component vector such that the rectangle
% [rect(1),rect(2)] x [rect(3),rect(4)] in 2d
% or [rect(1),rect(2)] x [rect(3),rect(4)] x [rect(5),rect(6)] in 3d
% contains the mesh


% OUTPUT:

% V = Chebyshev-Vandermonde matrix


% FUNCTION BODY

if length(gmesh(1,:)) == 2
    
    % rectangle containing the mesh
    if isempty(rect)
        rect=[min(gmesh(:,1)) max(gmesh(:,1)) min(gmesh(:,2)) max(gmesh(:,2))];
    end;
    
    % couples with length less or equal to deg
    % graded lexicographical order
    j=(0:1:deg);
    [j1,j2]=meshgrid(j);
    dim=(deg+1)*(deg+2)/2;
    couples=zeros(dim,2);
    for s=0:deg
        good=find(j1(:)+j2(:)==s);
        couples(1+s*(s+1)/2:(s+1)*(s+2)/2,:)=[j1(good) j2(good)];
    end;
    
    % mapping the mesh in the square [-1,1]^2
    a=rect(1);b=rect(2);c=rect(3);d=rect(4);
    map=[(2*gmesh(:,1)-b-a)/(b-a) (2*gmesh(:,2)-d-c)/(d-c)];
    
    % Chebyshev-Vandermonde matrix on the mesh
    T1=chebpolys(deg,map(:,1));
    T2=chebpolys(deg,map(:,2));
    V=T1(:,couples(:,1)+1).*T2(:,couples(:,2)+1);
    
end;


if length(gmesh(1,:)) == 3
    
    % parallelepiped containing the mesh
    if isempty(rect)
        rect=[min(gmesh(:,1)) max(gmesh(:,1)) min(gmesh(:,2)) max(gmesh(:,2)) ...
            min(gmesh(:,3)) max(gmesh(:,3))];
    end;
    
    % triples with length less or equal to deg
    % graded lexicographical order
    j=(0:1:deg);
    [j1,j2,j3]=meshgrid(j,j,j);
    dim=(deg+1)*(deg+2)*(deg+3)/6;
    triples=zeros(dim,3);
    for s=0:deg
        good=find(j1(:)+j2(:)+j3(:)==s);
        triples(1+s*(s+1)*(s+2)/6:(s+1)*(s+2)*(s+3)/6,:)=[j1(good) j2(good)
            j3(good)];
    end;
    
    %mapping the mesh in the cube [-1,1]^3
    a=rect(1);b=rect(2);
    c=rect(3);d=rect(4);
    e=rect(5);f=rect(6);
    map=[(2*gmesh(:,1)-b-a)/(b-a) (2*gmesh(:,2)-d-c)/(d-c) ...
        (2*gmesh(:,3)-f-e)/(f-e)];
    
    %Chebyshev-Vandermonde matrix on the mesh
    T1=chebpolys(deg,map(:,1));
    T2=chebpolys(deg,map(:,2));
    T3=chebpolys(deg,map(:,3));
    V=T1(:,triples(:,1)+1).*T2(:,triples(:,2)+1).*T3(:,triples(:,3)+1);
    
end;








function T=chebpolys(deg,x)

% computes the Chebyshev-Vandermonde matrix on the real line by recurrence

% INPUT:
% deg = maximum polynomial degree
% x = 1-column array of abscissas

% OUTPUT
% T: Chebyshev-Vandermonde matrix at x, T(i,j+1)=T_j(x_i), j=0,...,deg

T=zeros(length(x),deg+1);
t0=ones(length(x),1);
T(:,1)=t0;
t1=x;
T(:,2)=t1;

for j=2:deg
    t2=2*x.*t1-t0;
    T(:,j+1)=t2;
    t0=t1;
    t1=t2;
end;








