
function [pts,stats_matrix]=pointset_AFP_disk(deg)
%--------------------------------------------------------------------------
% OBJECT:
%--------------------------------------------------------------------------
% INPUT:
% deg: DEGREE OF THE INTERPOLATORY SET.
%--------------------------------------------------------------------------
% OUTPUT:
%  pts: MATRIX N x 2 WHOSE COMPONENTS ARE THE CARTESIAN COORDINATES OF
%       EACH POINT IN THE POINTSET.
%
%  stats_matrix: MATRIX WITH STATISTICS WITH COLUMNS THAT CONSIST OF
%       DEGREE, CARDINALITY, LEBESGUE CONSTANT, ABS. VALUE OF VANDERMONDE
%       MATRIX "V" DETERMINANT, CONDITIONING OF "V", POINTS OUT OF THE
%       DOMAIN.
%--------------------------------------------------------------------------
% Copyright (C) 2018
% Gerard Meurant, Alvise Sommariva.
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation; either version 2 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%
% Authors:
% * Gerard Meurant    <gerard.meurant@gmail.com>
% * Alvise Sommariva  <alvise@math.unipd.it>
%
% Date: February 10, 2018.
%--------------------------------------------------------------------------
% STATISTICS.
%--------------------------------------------------------------------------
% |DEGR|  Card |  Leb. Const. |    Abs.Det.   |  Cond.Vand. | Out |
%--------------------------------------------------------------------------
%   1  |     3 | 1.795357e+00 | 1.834190e+00  | 3.02277e+00 |   0 |
%   2  |     6 | 2.157895e+00 | 3.678856e+01  | 7.80533e+00 |   0 |
%   3  |    10 | 2.763623e+00 | 4.715973e+03  | 1.48816e+01 |   0 |
%   4  |    15 | 3.382679e+00 | 6.668412e+06  | 2.58673e+01 |   0 |
%   5  |    21 | 3.881098e+00 | 8.407747e+10  | 3.41588e+01 |   0 |
%   6  |    28 | 4.842268e+00 | 1.688907e+16  | 4.98705e+01 |   0 |
%   7  |    36 | 5.488209e+00 | 4.086128e+22  | 6.15348e+01 |   0 |
%   8  |    45 | 6.646185e+00 | 1.812212e+30  | 8.28745e+01 |   0 |
%   9  |    55 | 8.069821e+00 | 1.021716e+39  | 9.05080e+01 |   0 |
%  10  |    66 | 7.805513e+00 | 2.427938e+49  | 1.15408e+02 |   0 |
%  11  |    78 | 1.179652e+01 | 6.022076e+60  | 1.43246e+02 |   0 |
%  12  |    91 | 2.053252e+01 | 5.077595e+73  | 1.47548e+02 |   0 |
%  13  |   105 | 2.105305e+01 | 1.439820e+88  | 2.70251e+02 |   0 |
%  14  |   120 | 1.728409e+01 | 8.457649e+103 | 2.11343e+02 |   0 |
%  15  |   136 | 1.664044e+01 | 6.546844e+121 | 2.28466e+02 |   0 |
%  16  |   153 | 2.409251e+01 | 2.590437e+140 | 2.76931e+02 |   0 |
%  17  |   171 | 2.886381e+01 | 1.077266e+160 | 4.04172e+02 |   0 |
%  18  |   190 | 3.235490e+01 | 9.956084e+181 | 3.82217e+02 |   0 |
%  19  |   210 | 3.771867e+01 | 2.070614e+205 | 6.06969e+02 |   0 |
%  20  |   231 | 3.760500e+01 | 2.258682e+230 | 7.05914e+02 |   0 |
%--------------------------------------------------------------------------
% CAPTION.
%--------------------------------------------------------------------------
% Card.      : cardinality of the point set
% Leb. Const.: Lebesgue Constant (evaluated on a WAM of degree  500.
% Abs.Det.   : Absolute value of the determinant of the Vandermonde matrix
%           (on the point set).
% Cond.Vand. : Conditioning of the Vandermonde matrix
%           (on the point set). We used Koornwinder IIb polynomial basis.
% Out        : Points outside the domain.
%
% The polynomial basis is the orthonormal Koornwinder type II.
% The cardinality of the WAM is 250501.
%--------------------------------------------------------------------------

switch deg
    case    1
        
        % DEGREE.             :           1
        % CARDIN.             :           3
        % LEBESGUE CONST.     : 1.79536e+00
        % ABS.DET.VAND.MATRIX : 1.83419e+00
        % CONDITIONING VAND.  : 3.02277e+00
        % POINTS OUT DISK     :           0
        
        pts=[
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            -2.61321923212860574015081738252775e-01 9.65251704193435644540954854164738e-01
            -5.95308864276666072079535751981894e-01 -8.03496954638675653370682994136587e-01
            ];
        
    case    2
        
        % DEGREE.             :           2
        % CARDIN.             :           6
        % LEBESGUE CONST.     : 2.15790e+00
        % ABS.DET.VAND.MATRIX : 3.67886e+01
        % CONDITIONING VAND.  : 7.80533e+00
        % POINTS OUT DISK     :           0
        
        pts=[
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            4.07683614941688809985009811498458e-01 9.13123250228618843493677559308708e-01
            -6.90420020571746673176960484852316e-01 7.23408733147249849437798729923088e-01
            -8.63421704896663477768470329465345e-01 -5.04482863448639795578287703392562e-01
            2.31176322114969662324668320252385e-01 -9.72911870671437473134801621199585e-01
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            ];
        
    case    3
        
        % DEGREE.             :           3
        % CARDIN.             :          10
        % LEBESGUE CONST.     : 2.76362e+00
        % ABS.DET.VAND.MATRIX : 4.71597e+03
        % CONDITIONING VAND.  : 1.48816e+01
        % POINTS OUT DISK     :           0
        
        pts=[
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            6.20009547460775078953076899779262e-01 7.84594265246365774046921615081374e-01
            -2.00807072855718332649033186498855e-01 9.79630807749081733426521623187000e-01
            -9.06672601177072490230557377799414e-01 4.21835032062062353919174029215355e-01
            -2.49911708978592279895014627300043e-02 5.35243678705893288771733296016464e-01
            4.00042461626853029788009052936104e-01 -2.68426212218608906123762380957487e-01
            -5.04767200155524231597325979237212e-01 -1.79778830412862883347813180989760e-01
            -8.63421704896663477768470329465345e-01 -5.04482863448639795578287703392562e-01
            -2.00807072855718665715940574045817e-01 -9.79630807749081622404219160671346e-01
            6.20009547460774967930774437263608e-01 -7.84594265246366107113829002628336e-01
            ];
        
    case    4
        
        % DEGREE.             :           4
        % CARDIN.             :          15
        % LEBESGUE CONST.     : 3.38268e+00
        % ABS.DET.VAND.MATRIX : 6.66841e+06
        % CONDITIONING VAND.  : 2.58673e+01
        % POINTS OUT DISK     :           0
        
        pts=[
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            7.54822939153257044253564345126506e-01 6.55928601699939939351224893471226e-01
            1.70243555722398631768754739823635e-01 9.85402015288681321081298847275320e-01
            -5.44204182756027110556829029519577e-01 8.38952804078301306311971075047040e-01
            -9.52013107532729829785012043430470e-01 3.06057254587887683783264947123826e-01
            -1.16539733372777357112148877149593e-01 6.74554097742164238127315911697224e-01
            -6.68384552106206508703678537131054e-01 1.47874374863020408010783057761728e-01
            5.99593349122257368577493252814747e-01 2.78964511911824919465630046033766e-01
            4.41483958274292098700186670612311e-01 -4.92367035659690310822611536423210e-01
            -3.72533398339925714282117041875608e-01 -5.74302714042559281359956457890803e-01
            -9.19353038982236303588990722346352e-01 -3.93433589966752372824032590870047e-01
            -4.63663531985327337370250688763917e-01 -8.86011359468314774190389471186791e-01
            2.00807072855718332649033186498855e-01 -9.79630807749081733426521623187000e-01
            7.74857107102889797189959608658683e-01 -6.32136427974326453593789665319491e-01
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            ];
        
    case    5
        
        % DEGREE.             :           5
        % CARDIN.             :          21
        % LEBESGUE CONST.     : 3.88110e+00
        % ABS.DET.VAND.MATRIX : 8.40775e+10
        % CONDITIONING VAND.  : 3.41588e+01
        % POINTS OUT DISK     :           0
        
        pts=[
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            8.63421704896663477768470329465345e-01 5.04482863448639795578287703392562e-01
            4.90994080973322200023289951786865e-01 8.71162908099950450413473390653962e-01
            -4.66403903874177058752614755121613e-02 9.98911744842610804795413059764542e-01
            -5.17849624898325422428513320483034e-01 8.55471662881163830149944260483608e-01
            -9.19353038982236192566688259830698e-01 3.93433589966752705890939978417009e-01
            7.17458830778288536578202183591202e-01 2.80968829999189595447717238130281e-01
            2.24384792262238424997988772702229e-01 7.37117441317388988331060772907222e-01
            -4.19316529587494712938422480874578e-01 6.46424245606213365711312235362129e-01
            -7.64557077063346635092955239088042e-01 9.56197323004731453144344754946360e-02
            -2.81283898278430366524105465941830e-01 1.88739892925144703283990565978456e-01
            2.87739339399498195160020941330004e-01 1.12683518646103264515545561152976e-01
            -8.85196447836241739892670921108220e-02 -3.26967354791707576211479135963600e-01
            -5.49055380965322248343341016152408e-01 -5.40581951165488905353129212016938e-01
            1.07498416226645465698652515129652e-01 -7.62977553930405671955838897702051e-01
            6.65277857722928689376828970125644e-01 -3.88710727040627090911328878064523e-01
            -9.82635248222263668615994447463891e-01 -1.85547753829493261568117645765597e-01
            -7.12583964147506909725393597909715e-01 -7.01586839984776666767629649257287e-01
            -1.70243555722398631768754739823635e-01 -9.85402015288681321081298847275320e-01
            3.50126449191390465287554434326012e-01 -9.36702444523675126220041420310736e-01
            7.94141650844753299587353012611857e-01 -6.07732703080531955208698491333053e-01
            ];
        
    case    6
        
        % DEGREE.             :           6
        % CARDIN.             :          28
        % LEBESGUE CONST.     : 4.84227e+00
        % ABS.DET.VAND.MATRIX : 1.68891e+16
        % CONDITIONING VAND.  : 4.98705e+01
        % POINTS OUT DISK     :           0
        
        pts=[
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            8.93115016186791565999669728626031e-01 4.49828376007635810651663632597774e-01
            5.95308864276666072079535751981894e-01 8.03496954638675653370682994136587e-01
            1.39515338943923122849710694026726e-01 9.90219910019669469214420587377390e-01
            -3.79088403840379450926434401480947e-01 9.25360460617241531444676638784586e-01
            -7.74857107102889797189959608658683e-01 6.32136427974326453593789665319491e-01
            -9.82635248222263668615994447463891e-01 1.85547753829493428101571339539078e-01
            7.38678080481006138491295587300556e-01 3.72044311553388962199306888578576e-01
            3.60511535149943440625719404124538e-01 7.44374710321468513463116778439144e-01
            -2.16134286321425672650065052948776e-01 7.98340933823806220281937839899911e-01
            -6.86797769655803524990744790557073e-01 4.60837389909014294442357595471549e-01
            4.62999456609476056634377982845763e-01 1.33109374913382677574347212612338e-01
            3.74244788498353173067378918403847e-02 4.80297835611737666816623004706344e-01
            -4.36792856824413766592130059507326e-01 2.03220576560713350744435956585221e-01
            -3.10302557243396781583300025886274e-01 -3.68509057526012928640568588889437e-01
            2.62172364504237476179326904457412e-01 -4.04168595762658289238800080056535e-01
            -8.23482234494308795724748506472679e-01 -7.70667620612089099285668680749950e-02
            -5.89364354285963676360893259698059e-01 -5.80268846518084080443600214493927e-01
            -6.42508010276486296596942793257767e-02 -8.24581173026430636063821566494880e-01
            5.12797852569568890501727764785755e-01 -6.48922675472492271886437720240792e-01
            8.12718525402096325649381469702348e-01 -1.53462942792652434764733015981619e-01
            -9.42034263469989197048448659188580e-01 -3.35516685797524916168299569108058e-01
            -7.12583964147506909725393597909715e-01 -7.01586839984776666767629649257287e-01
            -3.50126449191390798354461821872974e-01 -9.36702444523675015197738957795082e-01
            1.70243555722398520746452277307981e-01 -9.85402015288681321081298847275320e-01
            6.20009547460774967930774437263608e-01 -7.84594265246366107113829002628336e-01
            8.93115016186791454977367266110377e-01 -4.49828376007636032696268557629082e-01
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            ];
        
    case    7
        
        % DEGREE.             :           7
        % CARDIN.             :          36
        % LEBESGUE CONST.     : 5.48821e+00
        % ABS.DET.VAND.MATRIX : 4.08613e+22
        % CONDITIONING VAND.  : 6.15348e+01
        % POINTS OUT DISK     :           0
        
        pts=[
            9.95649347969018605475355343514821e-01 9.31792674840715706707783283491153e-02
            8.47314704957777498073312472115504e-01 5.31091132257275000405627451982582e-01
            5.44204182756027332601433954550885e-01 8.38952804078301195289668612531386e-01
            2.00807072855718665715940574045817e-01 9.79630807749081622404219160671346e-01
            -2.00807072855718332649033186498855e-01 9.79630807749081733426521623187000e-01
            -6.20009547460774967930774437263608e-01 7.84594265246366107113829002628336e-01
            -8.63421704896663477768470329465345e-01 5.04482863448639795578287703392562e-01
            -9.92269872363276461690873020415893e-01 1.24098752613259658006406027652702e-01
            8.40419409001125106151164345646976e-01 1.85935617957015464352110711843125e-01
            5.94273527991029726358362950122682e-01 6.22668299321518792233121075696545e-01
            1.72843086926559863503172209675540e-01 8.43209407177455183557412965456024e-01
            -3.50910421101204350602387194157927e-01 7.85963557306210591768547146784840e-01
            -6.99488820009324219206803263659822e-01 5.01589700584871289734678612148855e-01
            -8.59077005159671336365079241659259e-01 5.35120197408727946153383925320668e-02
            5.40382157964025777729943911253940e-01 2.31254461938941002774328126179171e-01
            1.53601172565225679855771545589960e-01 5.67360716475078352694083605456399e-01
            -3.35056554709036502615049357700627e-01 4.82937478312707557659422263895976e-01
            -5.85228003190806034261584045452764e-01 5.47693992465529411184377295285231e-02
            1.35338076804238427452986570642679e-01 2.08639078182871195998870916810120e-01
            -2.42818174828921318209040691726841e-01 5.37214477729293302110669117155339e-02
            8.26955731895818169085998761147493e-02 -2.01861130339242522691023395964294e-01
            -4.05818705979538507655490775505314e-01 -4.25208984723534522132126767246518e-01
            8.20050586998236769042947003072186e-02 -5.82036659635941311385920471366262e-01
            5.07506544647482571441798882005969e-01 -2.96527587169387984999957552645355e-01
            -7.43183348431955104196333650179440e-01 -4.34229602473535980511343268517521e-01
            -3.99094688368858585558740514898091e-01 -7.62627213497277134379714880196843e-01
            9.35214719029079238898916059952171e-02 -8.55646288687078593326873487967532e-01
            5.54412901166159599597449414432049e-01 -6.58409577748820473708235567755764e-01
            8.10848481446325597410407226561802e-01 -2.88793312227006992998212808743119e-01
            -9.52013107532729940807314505946124e-01 -3.06057254587887295205206328319036e-01
            -7.12583964147506909725393597909715e-01 -7.01586839984776666767629649257287e-01
            -3.20825769815367811688133770076092e-01 -9.47138229311000112708995857246919e-01
            4.66403903874177058752614755121613e-02 -9.98911744842610804795413059764542e-01
            4.07683614941688698962707348982804e-01 -9.13123250228618843493677559308708e-01
            7.34058528759460204859976784064202e-01 -6.79086206865885944061744794453261e-01
            9.31144062097659652721404199837707e-01 -3.64651526282654958599493966175942e-01
            ];
        
    case    8
        
        % DEGREE.             :           8
        % CARDIN.             :          45
        % LEBESGUE CONST.     : 6.64619e+00
        % ABS.DET.VAND.MATRIX : 1.81221e+30
        % CONDITIONING VAND.  : 8.28745e+01
        % POINTS OUT DISK     :           0
        
        pts=[
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            9.42034263469989197048448659188580e-01 3.35516685797524916168299569108058e-01
            7.54822939153257044253564345126506e-01 6.55928601699939939351224893471226e-01
            4.35884418475371182566391325963195e-01 9.00002652068530029083603949402459e-01
            4.66403903874176087307468208109640e-02 9.98911744842610804795413059764542e-01
            -2.61321923212860574015081738252775e-01 9.65251704193435644540954854164738e-01
            -5.70032258213782561107052515581017e-01 8.21622312620400530036590680538211e-01
            -8.47314704957777609095614934631158e-01 5.31091132257275000405627451982582e-01
            -9.87930439740756671795907095656730e-01 1.54898180214084690842213376527070e-01
            8.87130064844290133763138328504283e-01 8.30233352474008118315751403315517e-02
            7.39881110251052631632262546190759e-01 4.96456008967467776749060703878058e-01
            4.13127232027148627580004358605947e-01 7.89441901791273714650287729455158e-01
            -4.15568921258816156849569267706102e-02 8.90036881743152541446306713623926e-01
            -4.13127232027148738602306821121601e-01 7.89441901791273714650287729455158e-01
            -7.07585392032396121031467828288442e-01 5.41493803407386176118620824126992e-01
            -8.80252467253297421478919204673730e-01 1.38015289155655029862401761420188e-01
            6.22978435974131161678712942375569e-01 2.21881165331970831022800894061220e-01
            3.24700211551339879001432109362213e-01 5.76110367756355379320609699789202e-01
            -9.22631490382672669881003457703628e-02 6.54844175775726400345888578158338e-01
            -4.71240230530110104645302726567024e-01 4.63967701836859147679348325254978e-01
            -6.60991977063876468001524244755274e-01 2.05667079289744078596235965505912e-02
            2.62319653052075663079989453763119e-01 2.58271341638852736100062656987575e-01
            -1.50078348387281707010387776790594e-01 3.36143088036389703354700486670481e-01
            -3.65278902906213454038208965357626e-01 -4.56837977944825751608348696208850e-02
            -6.26708327977570089917236373366904e-02 -3.62750676092725821142437325761421e-01
            3.28777565829731333035823581667501e-01 -1.65592869702686051880391460144892e-01
            -5.12422198857500732493974737735698e-01 -4.18039320322731944301608564273920e-01
            -1.12584083394110481823879865714844e-01 -6.51658044824243387616036216059001e-01
            3.06626195219769526190134456555825e-01 -5.85929824827935763487118947523413e-01
            6.15776616590845482512861508439528e-01 -2.41148381039099163336558717674052e-01
            5.85011946789474879260239958966849e-01 -6.52437291701728927506565014482476e-01
            -8.39358674760744216669650086259935e-01 -2.98947556019653393200030677689938e-01
            -5.94825388403371735357438865321456e-01 -6.63381778056206794502713819383644e-01
            -1.78920412017614216537708671239670e-01 -8.72857441000352340942924911360024e-01
            2.59474208534119510360937965742778e-01 -8.52388269072157478412066211603815e-01
            8.07851202951609614899552980205044e-01 -3.75857765698506873786044479857082e-01
            -9.82635248222263668615994447463891e-01 -1.85547753829493261568117645765597e-01
            -8.78693400099268773217886518978048e-01 -4.77386540051127017392218476743437e-01
            -6.20009547460775078953076899779262e-01 -7.84594265246365774046921615081374e-01
            -2.61321923212860740548535432026256e-01 -9.65251704193435644540954854164738e-01
            1.08652150085474366592031003619923e-01 -9.94079830940052588950095469044754e-01
            4.35884418475371071544088863447541e-01 -9.00002652068530029083603949402459e-01
            6.90420020571746673176960484852316e-01 -7.23408733147249849437798729923088e-01
            9.19353038982236192566688259830698e-01 -3.93433589966752705890939978417009e-01
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            ];
        
    case    9
        
        % DEGREE.             :           9
        % CARDIN.             :          55
        % LEBESGUE CONST.     : 8.06982e+00
        % ABS.DET.VAND.MATRIX : 1.02172e+39
        % CONDITIONING VAND.  : 9.05080e+01
        % POINTS OUT DISK     :           0
        
        pts=[
            9.87930439740756671795907095656730e-01 1.54898180214084663086637760898157e-01
            8.47314704957777498073312472115504e-01 5.31091132257275000405627451982582e-01
            5.70032258213782783151657440612325e-01 8.21622312620400530036590680538211e-01
            2.31176322114969773346970782768039e-01 9.72911870671437362112499158683931e-01
            -1.39515338943923011827408231511072e-01 9.90219910019669469214420587377390e-01
            -4.90994080973322089000987489271211e-01 8.71162908099950561435775853169616e-01
            -7.34058528759460204859976784064202e-01 6.79086206865885944061744794453261e-01
            -9.19353038982236192566688259830698e-01 3.93433589966752705890939978417009e-01
            -9.99516282291988078512190440960694e-01 3.10998622698367525873930361512976e-02
            6.82510359306270797397075966728153e-01 6.31398387034945662854568126931554e-01
            9.17754625683981140049638725031400e-01 0.00000000000000000000000000000000e+00
            8.43740504162572846347245558717987e-01 3.61075497091441766084329856312252e-01
            4.00034941319344083954234747579903e-01 8.25981597063744077402702714607585e-01
            4.28044340217592009234515160187584e-02 9.16755874479362797835335641138954e-01
            -5.69016630150358082573802676051855e-01 7.20065016214976982844575559283840e-01
            -8.19660437373251560266851356573170e-01 4.12832072844921127430239948807866e-01
            -2.63498949699324469531092063334654e-01 8.65609784130065063045833539945306e-01
            -9.00890464812521329918126866687089e-01 8.43111219485554314045927526422020e-02
            7.27558508419007821110824352217605e-01 4.53197152657131702158110897471488e-02
            5.64846521812519375416172806581017e-01 4.60807624243518676454556270982721e-01
            2.12286389057221441811762474571879e-01 6.97373464354432370804204310843488e-01
            -4.69536282542837080100639468582813e-01 5.57611817611956439577625133097172e-01
            -7.00590564277063454490246385830687e-01 2.01415294877530570705204127079924e-01
            -1.41992042926499528743278233378078e-01 6.92703587218630789656970137002645e-01
            4.08196972233820065678600030878442e-01 2.55855104247782183524151378151146e-01
            -3.63638724233336874114286274561891e-01 3.15996013817346654839468556019710e-01
            4.93270439150808107098278298963123e-02 4.51302799229478746045884918203228e-01
            1.72269581012872485814568790374324e-01 7.37221032901510670010125636508747e-02
            -1.58770943282803861373864151573798e-01 9.95165545271891882617154578838381e-02
            2.91411896182525078508041005420637e-03 -1.87358653300429695010009822908614e-01
            -4.75939119102032903718679790472379e-01 -7.46227674298049342693772700840782e-02
            -2.36538202471093139678259831271134e-01 -4.19685931718286220348801407453720e-01
            2.36538202471093084167108600013307e-01 -4.19685931718286275859952638711547e-01
            4.62999456609476334190134139134898e-01 -1.33109374913382871863376522014732e-01
            -7.68286721159375241008149259869242e-02 -7.02920589498487902169188146217493e-01
            -6.93987688285326642656514195550699e-01 -2.23106136789297637701423582257121e-01
            -4.33961485683560521664503539795987e-01 -5.85724072160239361828359960782109e-01
            2.97188565206241983585044863502844e-01 -6.65638202385734234489689242764143e-01
            6.29407335104375631473772045865189e-01 -3.67752180525778238351364279878908e-01
            1.40717001403967995748489627771960e-02 -9.04717625632165622917568725824822e-01
            6.44765247914140249285708250681637e-01 -6.34814752472374288494449956488097e-01
            -8.82027301161382948357925215532305e-01 -2.53577193316573523684809288170072e-01
            -6.73685610291771985735920225124573e-01 -6.23234507589355612999781897087814e-01
            -3.21330168319706233059918076833128e-01 -8.59663001351095301671989545866381e-01
            3.74153523428301715281207862062729e-01 -8.38023086716906173698760085244430e-01
            8.43740504162572513280338171171024e-01 -3.61075497091441932617783550085733e-01
            -9.69198999199666122450480543193407e-01 -2.46278906832001348581329125408956e-01
            -8.63421704896663477768470329465345e-01 -5.04482863448639795578287703392562e-01
            -6.20009547460775078953076899779262e-01 -7.84594265246365774046921615081374e-01
            -3.50126449191390798354461821872974e-01 -9.36702444523675015197738957795082e-01
            -4.66403903874176087307468208109640e-02 -9.98911744842610804795413059764542e-01
            2.91214712227251926179860674892552e-01 -9.56657719031419717659048274072120e-01
            5.95308864276666072079535751981894e-01 -8.03496954638675653370682994136587e-01
            8.30387982764797305534898441692349e-01 -5.57185604695428882138230619602837e-01
            9.76389420563607002279127300425898e-01 -2.16017821976483392898416013849783e-01
            ];
        
    case   10
        
        % DEGREE.             :          10
        % CARDIN.             :          66
        % LEBESGUE CONST.     : 7.80551e+00
        % ABS.DET.VAND.MATRIX : 2.42794e+49
        % CONDITIONING VAND.  : 1.15408e+02
        % POINTS OUT DISK     :           0
        
        pts=[
            9.87930439740756671795907095656730e-01 1.54898180214084663086637760898157e-01
            9.19353038982236303588990722346352e-01 3.93433589966752372824032590870047e-01
            7.34058528759460093837674321548548e-01 6.79086206865885944061744794453261e-01
            4.63663531985327337370250688763917e-01 8.86011359468314774190389471186791e-01
            1.39515338943923122849710694026726e-01 9.90219910019669469214420587377390e-01
            -7.76838472890061110254222853654937e-02 9.96978043825629267615795470192097e-01
            -2.91214712227251926179860674892552e-01 9.56657719031419717659048274072120e-01
            -5.95308864276666072079535751981894e-01 8.03496954638675653370682994136587e-01
            -8.30387982764797305534898441692349e-01 5.57185604695428882138230619602837e-01
            -9.61070940398724538056285382481292e-01 2.76301732750830475460190882586176e-01
            9.18554492564195745885058386193123e-01 1.44020685669936726602813337194675e-01
            8.02789198618462651602101232128916e-01 4.69056303968118926128738621628145e-01
            5.53504183845284658538332678290317e-01 7.47072574905859654315065654373029e-01
            1.58288454984692517069078121494385e-01 9.16203622962311170496718659705948e-01
            -2.70764591773612728342612854248728e-01 8.89477852198903629066251141921384e-01
            -5.98878714882386531392910455906531e-01 7.11216281149079865286921631195582e-01
            -8.30397341244183739839002100779908e-01 4.18239846697198880409729326856905e-01
            -9.27977923588832687151750633347547e-01 5.78038670199886583933590600281605e-02
            2.99538602427975642061852568076574e-01 7.31178206210991055158388007839676e-01
            -3.68531382437770685389644143015175e-02 7.89295122108335633903664074750850e-01
            -3.66366063854022760981621331666247e-01 7.00086316705688549433261869126000e-01
            7.17458830778288536578202183591202e-01 2.80968829999189595447717238130281e-01
            5.49055380965322248343341016152408e-01 5.40581951165489016375431674532592e-01
            -5.97037662281714620604589072172530e-01 4.87069488995202415360097347729607e-01
            -7.40517886824198501294347352086334e-01 2.12894144086411885030685198216815e-01
            1.46884709306299876541146431918605e-01 5.42551938370571917857887456193566e-01
            -2.45003386289748242843344883112877e-01 5.05876530750555275517399422824383e-01
            4.35443885380908946025613204255933e-01 3.12248261647429248277774149755714e-01
            -4.92613992332052108569229176282533e-01 2.10812259548965791333685615427385e-01
            1.51725515158265600312859078258043e-01 2.69204143471985235525067992057302e-01
            -1.76149655129987064983154709807422e-01 2.53895257557349707244753744816990e-01
            -2.52953591817856371370965007372433e-01 -1.17688222161088443251664159561187e-01
            3.35753608517889434725844921558746e-02 -3.07187561525850638677326287506730e-01
            3.05287295140208414512272838692297e-01 -4.78661700839053544376255899805983e-02
            -5.10114132187257762751642076182179e-01 -1.63993677805898546706586671461992e-01
            -2.91074666382309554801821604996803e-01 -4.80846401929023781196548270600033e-01
            6.10715675309401045955937092912791e-02 -5.58755749229459808269382392609259e-01
            4.00531601571292250607569940257235e-01 -3.94350300875245585174866391753312e-01
            5.60996082154854525114728858170565e-01 -3.49445198073288360030019816804270e-02
            -7.61213486761561752835802963090828e-01 -1.19351099136822957902381858730223e-01
            -6.52867200968628158541662287461804e-01 -4.09212750525018487124384591879789e-01
            6.88156947293836140921996502584079e-01 -3.46598720690210637673800420088810e-01
            7.69022759750360163089055731688859e-01 -4.79025289395251449575674485004129e-02
            -4.50413846043454302314756887426483e-01 -6.49208988596715941277182082558284e-01
            -1.34518798878713430333675660222070e-01 -7.78620341585458231037364384974353e-01
            2.06484827470297060969528502027970e-01 -7.62698472272620442424795328406617e-01
            5.08947070130900014461872160609346e-01 -6.04415273952933929102471211081138e-01
            -8.75881307075453641175499797100201e-01 -3.11955525077695305924407875863835e-01
            -6.82510359306270686374773504212499e-01 -6.31398387034945551832265664415900e-01
            -3.79054638854702419248354772207676e-01 -8.49000526780423592221325179707492e-01
            4.33651382748694094737551552043442e-02 -9.28764651832264309661013612640090e-01
            4.56514751199306956230827836407116e-01 -8.09986787329361690446205557236681e-01
            7.38374233419929359989453132584458e-01 -5.65055577029585087700525036780164e-01
            9.01138439602275576056911177147413e-01 -2.28984336542658706292741044308059e-01
            -9.99516282291988078512190440960694e-01 -3.10998622698369156513997779711644e-02
            -9.52013107532729940807314505946124e-01 -3.06057254587887295205206328319036e-01
            -8.47314704957777498073312472115504e-01 -5.31091132257275000405627451982582e-01
            -6.20009547460775078953076899779262e-01 -7.84594265246365774046921615081374e-01
            -3.20825769815367811688133770076092e-01 -9.47138229311000112708995857246919e-01
            1.55518119203506659242730592040971e-02 -9.99879063260149303360435624199454e-01
            2.31176322114969662324668320252385e-01 -9.72911870671437473134801621199585e-01
            4.63663531985327448392553151279571e-01 -8.86011359468314774190389471186791e-01
            7.34058528759460204859976784064202e-01 -6.79086206865885944061744794453261e-01
            9.19353038982236192566688259830698e-01 -3.93433589966752705890939978417009e-01
            9.98065597133594328838057663233485e-01 -6.21696374314805247363047158160043e-02
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            ];
        
    case   11
        
        % DEGREE.             :          11
        % CARDIN.             :          78
        % LEBESGUE CONST.     : 1.17965e+01
        % ABS.DET.VAND.MATRIX : 6.02208e+60
        % CONDITIONING VAND.  : 1.43246e+02
        % POINTS OUT DISK     :           0
        
        pts=[
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            9.76389420563607002279127300425898e-01 2.16017821976483503920718476365437e-01
            9.06672601177072601252859840315068e-01 4.21835032062062242896871566699701e-01
            7.34058528759460093837674321548548e-01 6.79086206865885944061744794453261e-01
            4.90994080973322200023289951786865e-01 8.71162908099950450413473390653962e-01
            2.00807072855718665715940574045817e-01 9.79630807749081622404219160671346e-01
            -1.55518119203506659242730592040971e-02 9.99879063260149303360435624199454e-01
            -2.61321923212860574015081738252775e-01 9.65251704193435644540954854164738e-01
            -5.44204182756027110556829029519577e-01 8.38952804078301306311971075047040e-01
            -7.54822939153256822208959420095198e-01 6.55928601699940050373527355986880e-01
            -9.06672601177072490230557377799414e-01 4.21835032062062353919174029215355e-01
            -9.92269872363276461690873020415893e-01 1.24098752613259658006406027652702e-01
            -4.14550716542525288232923230680171e-01 8.55953386932695292621531280019553e-01
            9.36787324125863407608960642392049e-01 8.76705808410047221501670833276876e-02
            8.40314743194393920511231499403948e-01 4.23234868315494838775236985384254e-01
            5.12031249939596810882846966705984e-01 7.89354559417495660333941032149596e-01
            1.88935513119935755144140898664773e-01 9.21715787686204879314288973546354e-01
            -1.60178887617591858560928130827961e-01 9.27145805873857864476406120957108e-01
            -6.49602919856937188747281197720440e-01 6.80641365111786522668069210340036e-01
            -8.53070814185211689917309740849305e-01 3.96896469338383572900141871286905e-01
            -9.39060726497814468594071968254866e-01 5.84942162721367939215433295885305e-02
            -3.91484068092108727920930277832667e-01 7.48084133111102334012798564799596e-01
            6.24299389990488506896326725836843e-01 5.42505804577096628626975416409550e-01
            7.70194782861682858410290464235004e-01 2.47605520213340674073876357397239e-01
            3.52637902129816604990253381401999e-01 7.28117440505963720021043172891950e-01
            1.25816801368866786753342168481140e-02 8.08919154497164116790486332320143e-01
            -6.26872567858446694266660870198393e-01 5.11409112994705039589860007254174e-01
            -7.77522723582480423942797642666847e-01 2.23532797370666863612242991621315e-01
            -1.72815288489860968690337017505954e-01 6.38332405006994840768186350032920e-01
            -3.93684815478506811015080302240676e-01 5.31362069853976382027838099020300e-01
            4.40090284115215224503003810241353e-01 4.61118080901765203805098281009123e-01
            1.47357333577952304137781425197318e-01 6.20157366277248822683532125665806e-01
            -5.60100252857117619775806360848946e-01 3.04297633011712032047313414295786e-01
            -6.36190954955804133241770159656880e-01 3.96284183328037969973323129124765e-02
            6.02264074803468774099712845782051e-01 1.13723527111562500158115085469035e-01
            2.65131221699641539402136913849972e-01 2.95688826836103424433588315878296e-01
            -7.97501054091932365386341530211212e-02 3.89058308898394045183266598542104e-01
            -3.42906108827995881327410643280018e-01 2.00354305080021333207440648038755e-01
            6.37757682090705907418382025753090e-02 1.42843947165310902747137333790306e-01
            -1.50344618426986015302659893677628e-01 -4.32231137525649553032813798836287e-02
            1.01973695167874273126429329749953e-01 -1.57204079303571941927941679750802e-01
            -3.69801900139186734417506841055001e-01 -1.44820584479909780784012696130958e-01
            -1.15655308682867913327996234329476e-01 -3.79934595172808975416245402811910e-01
            3.95420038357790493144250376644777e-01 -3.70059495321930845279645438949956e-02
            2.42707931071502985842514021896932e-01 -3.49829766201570913075613589171553e-01
            -5.86017682095639758088623239018489e-01 -2.50784008617757347980870008541388e-01
            -3.46888801410272062941686499470961e-01 -5.34768643586441605464187887264416e-01
            -2.97297037241841706489697827464624e-02 -6.36730309804402150319901920738630e-01
            6.06835993296561015952761408698279e-01 -1.95088236310941853801637080323417e-01
            2.88255537847462828260347578179790e-01 -5.95182432635673053233915652526775e-01
            4.85441614910654073433704525086796e-01 -4.49087766178042246956181315908907e-01
            -8.02763189748150529645442929904675e-01 -1.00397999844859212026371153569926e-01
            -1.87025573288107510583344605947786e-01 -7.87102237402313775760376302059740e-01
            1.37729929762238717527722542399715e-01 -7.97206976659864841039393468236085e-01
            7.10877893525413528941214735823451e-01 -3.86213823787218191174730463899323e-01
            8.05497242945271474390267485432560e-01 -7.53836109180229529957983913845965e-02
            -7.14119319527053542095984539628262e-01 -4.17247976412776278021254938721540e-01
            -4.92368397336682084297621031510062e-01 -6.64556722670417321374713992554462e-01
            5.02635898407677972343776673369575e-01 -6.78414916857259520099887595279142e-01
            -8.95730824669896619205644583416870e-01 -2.87963385040670538650431353744352e-01
            -6.90661552996608252286137030750979e-01 -6.38939152502183027593218866968527e-01
            -3.83581673116374533005767943905084e-01 -8.59140105825084487989329318224918e-01
            -4.38830463720387420334390071730013e-02 -9.39856850604922744452096594613977e-01
            3.01858797004214474135608270444209e-01 -8.91144145500077256016879800881725e-01
            7.47192607105389483557189578277757e-01 -5.71804012993040844392567123577464e-01
            9.04253165421912474819521321478533e-01 -2.59966986773986274972259025162202e-01
            5.42132893672659599282326325919712e-01 -7.81409254351125848181425226357533e-01
            -9.95649347969018605475355343514821e-01 -9.31792674840715706707783283491153e-02
            -9.61070940398724538056285382481292e-01 -2.76301732750830253415585957554867e-01
            -8.30387982764797527579503366723657e-01 -5.57185604695428549071323232055875e-01
            -6.20009547460775078953076899779262e-01 -7.84594265246365774046921615081374e-01
            -3.79088403840379339904131938965293e-01 -9.25360460617241531444676638784586e-01
            -1.08652150085474491492121273950033e-01 -9.94079830940052588950095469044754e-01
            1.39515338943923011827408231511072e-01 -9.90219910019669469214420587377390e-01
            4.07683614941688698962707348982804e-01 -9.13123250228618843493677559308708e-01
            6.44110411450397712940230121603236e-01 -7.64932531574647689431856178998714e-01
            8.12657913828250144661069498397410e-01 -5.82741036046301008610726057668217e-01
            9.52013107532729829785012043430470e-01 -3.06057254587887683783264947123826e-01
            ];
        
    case   12
        
        % DEGREE.             :          12
        % CARDIN.             :          91
        % LEBESGUE CONST.     : 2.05325e+01
        % ABS.DET.VAND.MATRIX : 5.07760e+73
        % CONDITIONING VAND.  : 1.47548e+02
        % POINTS OUT DISK     :           0
        
        pts=[
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            9.69198999199666122450480543193407e-01 2.46278906832001348581329125408956e-01
            8.63421704896663477768470329465345e-01 5.04482863448639795578287703392562e-01
            6.90420020571746673176960484852316e-01 7.23408733147249849437798729923088e-01
            5.44204182756027332601433954550885e-01 8.38952804078301195289668612531386e-01
            3.50126449191390798354461821872974e-01 9.36702444523675015197738957795082e-01
            7.76838472890062359255125556956045e-02 9.96978043825629267615795470192097e-01
            -2.00807072855718332649033186498855e-01 9.79630807749081733426521623187000e-01
            -4.35884418475371071544088863447541e-01 9.00002652068530029083603949402459e-01
            -6.67588140216153647088503930717707e-01 7.44530775081015172567333593178773e-01
            -8.63421704896663477768470329465345e-01 5.04482863448639795578287703392562e-01
            -9.82635248222263668615994447463891e-01 1.85547753829493428101571339539078e-01
            9.49216789843918440361392185877776e-01 5.91268387949166474015783023787662e-02
            8.62296885495731801896113211114425e-01 4.01188956044199296346874916707748e-01
            6.56628459545393616814124015945708e-01 6.88002589604513770105143066757591e-01
            3.87730158577057337776494705394725e-01 8.68431817310537979004436692775926e-01
            7.38817291750869797883893852485926e-02 9.48182465183559863852735816180939e-01
            -2.76961649704745849209786001665634e-01 9.09835557548889783952006382605759e-01
            -5.42132893672659599282326325919712e-01 7.81409254351125848181425226357533e-01
            -7.55273591897293261610002446104772e-01 5.77988147430407606108815343759488e-01
            -8.85570627867518278542036114231450e-01 3.46804210248092437129940890372382e-01
            -9.46918800330955878763461441849358e-01 8.86187495243355161100851091759978e-02
            8.18321780392883280086380182183348e-01 2.07940158500367733296343430993147e-01
            6.86149770525401936360765375866322e-01 4.92024530069868126780363581929123e-01
            4.81294154046872202368234638925060e-01 6.93718662760950843804153009841684e-01
            1.95188624476518052253126711548248e-01 8.21456661460299497612425057013752e-01
            -9.17380444830021568591860159358475e-02 8.39329361441008447997091934666969e-01
            -3.68029386809962277826713261674740e-01 7.59897372167333773340658353845356e-01
            -6.19786114784536268373926759522874e-01 5.73371448280105777683957057888620e-01
            -8.11459033367125881497372574813198e-01 2.33289268826120860245154631229525e-01
            7.68286721159375379786027338013810e-02 7.02920589498488013191490608733147e-01
            -6.10531342556080880434876689832890e-01 3.56723253736940304481350949572516e-01
            6.83222916044790573586453774623806e-01 4.25580453803558675374141273550777e-02
            6.11379299592122116280279442435130e-01 3.07928712960629069606000030034920e-01
            4.72625026957614935430029845520039e-01 4.95207354679488953586030675069196e-01
            2.79078638742878970546712480427232e-01 6.25075878300198839454537846904714e-01
            -1.37461900573393330793336986062059e-01 6.70603434323217562074148645478999e-01
            -4.07516960174286291263001658080611e-01 5.50031514920420350200913617300102e-01
            -6.81568879671977656542480872303713e-01 6.37855978887764546447769475889800e-02
            5.53084443038317496998956812603865e-02 5.06027804510614021005210361181526e-01
            3.91500935774594660543357349524740e-01 2.80737635165145438254086229790119e-01
            -1.96403079369227434591849146272580e-01 4.39900480705336938136440494417911e-01
            -4.08196972233820121189751262136269e-01 2.55855104247782183524151378151146e-01
            8.95075363666767154224856994915172e-02 2.64243142167513322959848665050231e-01
            -1.97494997541493161330450334389752e-01 1.51136977355489671159816111867258e-01
            2.48569591462629718714438809001877e-01 -7.73422123872822848272834406202492e-03
            -2.31670861753235229896219493639364e-01 -1.55449828123114336397847523585369e-01
            6.44961377969364979545474625410861e-02 -2.71433758877319797342408946860814e-01
            -4.80821768413631611682745869984501e-01 -2.99504512501872766083721444374532e-02
            -1.82627231359457603554474758311699e-01 -4.45795801770811850328612990779220e-01
            3.43289942834026529094870738845202e-01 -3.37992037864078498365927316626767e-01
            4.78029656711448647232742814594531e-01 -5.97850300228777026267934502357093e-02
            -4.13676534975875809241330216536880e-01 -2.96639322004847105773706061881967e-01
            8.66610206273069755278370962514600e-02 -5.01610436945819415655023476574570e-01
            -6.44866828744044995325168656563619e-01 -2.29676976253480852330213224377076e-01
            -3.90213932646239991530734414482140e-01 -5.62439176270731477913500384602230e-01
            -1.16539733372777440378875724036334e-01 -6.74554097742164238127315911697224e-01
            3.17399528945229880694256507922546e-01 -6.06516511943977931764493405353278e-01
            5.33740418870428556807894437952200e-01 -4.63811562236237540091110531648155e-01
            6.66118815809704001296154274314176e-01 -2.37246123728666130636710818180291e-01
            -6.22883217959981805300628820987185e-01 -4.17951089920782281605937669155537e-01
            -8.40654548498115161692112451419234e-01 -7.86738576146234908081922299061262e-02
            -5.82940903694401324486307203187607e-01 -6.10794194948258151178777097811690e-01
            -3.20074925596417536599602726710145e-01 -7.81307678054544485490851002396084e-01
            -3.93797840604124260721619066316634e-02 -8.43409081282559536241194564354373e-01
            2.45880713850501864126130158183514e-01 -8.07732827325287949804533127462491e-01
            5.02635898407677972343776673369575e-01 -6.78414916857259520099887595279142e-01
            8.37801162870634330559482805256266e-01 -1.04780042351341265427500104578939e-01
            -7.80411212566091805342693987768143e-01 -3.63091140558373048907725433309679e-01
            7.43183348431955104196333650179440e-01 -4.34229602473535980511343268517521e-01
            -9.28601520868667651420480524393497e-01 -2.05445157226621044621595046919538e-01
            -7.55273591897293483654607371136080e-01 -5.77988147430407273041907956212526e-01
            -5.17568934205198449838292162894504e-01 -7.97891531182759572438101258740062e-01
            -2.19861747560589404493214260583045e-01 -9.25294174382978118487130814173725e-01
            1.03334335348269412691024626838043e-01 -9.45426100933121627001298747927649e-01
            4.14550716542525288232923230680171e-01 -8.55953386932695292621531280019553e-01
            6.77707622509918450326438232877990e-01 -6.67248735914446999828442130819894e-01
            9.14032780488118090644888980023097e-01 -2.62778563396319386757937763832160e-01
            -9.95649347969018605475355343514821e-01 -9.31792674840715706707783283491153e-02
            -9.31144062097659541699101737322053e-01 -3.64651526282655180644098891207250e-01
            -8.12657913828250366705674423428718e-01 -5.82741036046300786566121132636908e-01
            -6.20009547460775078953076899779262e-01 -7.84594265246365774046921615081374e-01
            -4.35884418475371182566391325963195e-01 -9.00002652068530029083603949402459e-01
            -2.31176322114969773346970782768039e-01 -9.72911870671437362112499158683931e-01
            4.66403903874177058752614755121613e-02 -9.98911744842610804795413059764542e-01
            2.91214712227251926179860674892552e-01 -9.56657719031419717659048274072120e-01
            5.44204182756027110556829029519577e-01 -8.38952804078301306311971075047040e-01
            7.34058528759460204859976784064202e-01 -6.79086206865885944061744794453261e-01
            8.63421704896663477768470329465345e-01 -5.04482863448639795578287703392562e-01
            9.69198999199666011428178080677753e-01 -2.46278906832001820426114591100486e-01
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            ];
        
    case   13
        
        % DEGREE.             :          13
        % CARDIN.             :         105
        % LEBESGUE CONST.     : 2.10531e+01
        % ABS.DET.VAND.MATRIX : 1.43982e+88
        % CONDITIONING VAND.  : 2.70251e+02
        % POINTS OUT DISK     :           0
        
        pts=[
            9.98065597133594328838057663233485e-01 6.21696374314805247363047158160043e-02
            9.52013107532729940807314505946124e-01 3.06057254587887295205206328319036e-01
            8.47314704957777498073312472115504e-01 5.31091132257275000405627451982582e-01
            6.90420020571746673176960484852316e-01 7.23408733147249849437798729923088e-01
            5.17849624898325311406210857967380e-01 8.55471662881163830149944260483608e-01
            3.20825769815367811688133770076092e-01 9.47138229311000112708995857246919e-01
            7.76838472890062359255125556956045e-02 9.96978043825629267615795470192097e-01
            -1.39515338943923011827408231511072e-01 9.90219910019669469214420587377390e-01
            -3.79088403840379450926434401480947e-01 9.25360460617241531444676638784586e-01
            -5.95308864276666072079535751981894e-01 8.03496954638675653370682994136587e-01
            -7.94141650844753299587353012611857e-01 6.07732703080531955208698491333053e-01
            -9.31144062097659652721404199837707e-01 3.64651526282654958599493966175942e-01
            -9.95649347969018605475355343514821e-01 9.31792674840717094486564064936829e-02
            -4.71498515663434869527037562875194e-01 8.36572239844345588721807871479541e-01
            -6.84289881285496059959427839203272e-01 6.73729412391420923178486646065721e-01
            -8.43803723761331614383607302443124e-01 4.58431280038260335896183050863328e-01
            -9.30715679095851844415676623611944e-01 2.36500079146191505286012102260429e-01
            9.43704728034454376484063686802983e-01 1.18024927336940504973838983460155e-01
            8.62296885495731801896113211114425e-01 4.01188956044199296346874916707748e-01
            6.77707622509918450326438232877990e-01 6.67248735914446999828442130819894e-01
            4.40970223463071908653887476248201e-01 8.42646876933868416159612024785019e-01
            1.61911243027044293718574863305548e-01 9.37173007810676916840009198494954e-01
            -1.32686972225744992393003940378549e-01 9.41755097989407263803229852783261e-01
            -2.81141765219421635446650498124654e-01 8.29983557270146055095949577662395e-01
            8.27234149333384460867080179014010e-01 2.37824513512651547486598246905487e-01
            6.99488820009324330229105726175476e-01 5.01589700584871178712376149633201e-01
            5.12407358530913126237749111169251e-01 6.91603597427189464852403943950776e-01
            2.76148223425980332557116980751744e-01 8.15241679350076187127172033797251e-01
            1.33860981159068352652763422838689e-02 8.60637931669345546126237422868144e-01
            -5.12407358530913126237749111169251e-01 6.91603597427189464852403943950776e-01
            -7.14749835484687401354619851190364e-01 4.79593066802961542904881753202062e-01
            -8.45795455360354497820196684187977e-01 1.59708749737226929354960702767130e-01
            5.50625428179120079974495638452936e-01 5.09390081003461281383692949020769e-01
            7.23333606888276858803976665512891e-01 9.04640973571969736477527135320997e-02
            -1.68520286224479537429132847137225e-01 7.09222230965355637799518717656611e-01
            -5.19451354267116416352223495778162e-01 5.11434795760628113647783266060287e-01
            -6.70179522942584204869831410178449e-01 2.86800744059542145336649809905794e-01
            -7.28616012387742650879829398036236e-01 2.26708239118378410137388812017889e-02
            6.21330061794057142421365824702661e-01 3.37563259717335306664409699806129e-01
            3.84810467976870185236037968934397e-01 5.93229216859235797265625933505362e-01
            1.20380372704617946699734432058904e-01 6.96784447205516577206196870974964e-01
            -2.88599079759015164636792860619607e-01 5.12056709725374092911920342885423e-01
            5.23175613871997069459496287890943e-01 1.15748137208002602194234498256264e-01
            3.93328228792179046546806375772576e-01 3.63872585739391518622198873345042e-01
            2.03125724443493999027410268354288e-01 4.95832929812824507731505718766130e-01
            -2.49911708978592279895014627300043e-02 5.35243678705893288771733296016464e-01
            -4.54013922696100025166998648273875e-01 2.84572859239182018864511292122188e-01
            -5.35567606069837887083906480256701e-01 1.66641395243348477972489263265743e-02
            -1.80746976429128791696854250403703e-01 3.20696455859777629360252149126609e-01
            2.18184621183046739689714854648628e-01 2.59111654873561858547503788940958e-01
            -3.15413803043713747431553429123596e-01 1.23521299627257788444900654667435e-01
            -2.18250074125990907447647515482458e-02 1.54904521896112590262362118664896e-01
            -1.10129485146054209576682580973284e-01 -5.98323987246829405717285510490910e-02
            1.18068200369061376830970289120160e-01 -4.20513911457824490769397129952267e-02
            -3.15413803043713636409250966607942e-01 -1.23521299627257816200476270296349e-01
            -8.85196447836241739892670921108220e-02 -3.26967354791707576211479135963600e-01
            2.18184621183046628667412392132974e-01 -2.59111654873561803036352557683131e-01
            3.38574066714893595619884081315831e-01 -1.05347026651995082785484569853907e-02
            -4.35443885380909223581369360545068e-01 -3.12248261647429414811227843529196e-01
            -2.33558550932942005085024561594764e-01 -4.82245536530477691972151887966902e-01
            2.49911708978592453367362224980752e-02 -5.35243678705893621838640683563426e-01
            3.05438557953331135674801544155343e-01 -4.40247250454620764337221316964133e-01
            5.04767200155524120575023516721558e-01 -1.79778830412862966614540027876501e-01
            -6.79579779157270302825111230049515e-01 -1.95374828881705236982568862913467e-01
            -4.03073675278046827941125229699537e-01 -5.80974708828058683529604877548991e-01
            -1.41992042926499750787883158409386e-01 -6.92703587218630567612365211971337e-01
            2.05919897797178813592111623620440e-01 -6.76459160401571724108293892641086e-01
            -6.05326788023288986373415809794096e-01 -4.06170825473795626958661841854337e-01
            3.39993813631146724052456420395174e-02 -7.28175323553045039659536996623501e-01
            4.51967508800651507350210067670560e-01 -5.71944604619354257657448670215672e-01
            6.17665837467049283482367627584608e-01 -3.87148773717268968042048982169945e-01
            7.06515664144786059708280845370609e-01 -1.79529596676169994617922043289582e-01
            -8.50353249241356357046583980263676e-01 -1.33327373616693417579526226290909e-01
            -5.94273527991029726358362950122682e-01 -6.22668299321518903255423538212199e-01
            -3.75184037897926259130798598562251e-01 -7.74670107050391631986485663219355e-01
            -1.20086715640734642551734623339144e-01 -8.52323892529993076827565801067976e-01
            1.46535783236856148592508475303475e-01 -8.48176928053350764891149538016180e-01
            3.99094688368858641069891746155918e-01 -7.62627213497277134379714880196843e-01
            6.31835025983950471228922651789617e-01 -5.84518038208162171720516653294908e-01
            8.59077005159671447387381704174913e-01 -5.35120197408728015542322964392952e-02
            -7.82642654731968812065190377325052e-01 -3.94187610768774066016817414492834e-01
            8.05635209378760097642668824846623e-01 -3.44768483041503759345403068437008e-01
            -7.36932900907815158220159901247825e-01 -6.01197469012525020914949891448487e-01
            -5.17568934205198449838292162894504e-01 -7.97891531182759572438101258740062e-01
            -2.76961649704746182276693389212596e-01 -9.09835557548889783952006382605759e-01
            1.47906520670461495009506691644674e-02 -9.50941498620659242391184307052754e-01
            3.05123438978314454050888571146061e-01 -9.00781984818480041532495761202881e-01
            5.66172374578590376081876911484869e-01 -7.64171014532423864551446968107484e-01
            7.72883604465182671994227803224931e-01 -5.54219659644423545685754106671084e-01
            9.34541655963089579195468559191795e-01 -1.76466400363468750578377353122050e-01
            -9.60293685676942954110302252956899e-01 -0.00000000000000000000000000000000e+00
            -9.22910355552496786124549998930888e-01 -2.65330809302220471668221080108196e-01
            -9.87930439740756671795907095656730e-01 -1.54898180214084663086637760898157e-01
            -9.19353038982236303588990722346352e-01 -3.93433589966752372824032590870047e-01
            -7.94141650844753521631957937643165e-01 -6.07732703080531622141791103786090e-01
            -6.20009547460775078953076899779262e-01 -7.84594265246365774046921615081374e-01
            -4.35884418475371182566391325963195e-01 -9.00002652068530029083603949402459e-01
            -2.00807072855718665715940574045817e-01 -9.79630807749081622404219160671346e-01
            1.55518119203506659242730592040971e-02 -9.99879063260149303360435624199454e-01
            2.31176322114969662324668320252385e-01 -9.72911870671437473134801621199585e-01
            4.63663531985327448392553151279571e-01 -8.86011359468314774190389471186791e-01
            6.67588140216153647088503930717707e-01 -7.44530775081015172567333593178773e-01
            8.30387982764797305534898441692349e-01 -5.57185604695428882138230619602837e-01
            9.31144062097659652721404199837707e-01 -3.64651526282654958599493966175942e-01
            9.82635248222263668615994447463891e-01 -1.85547753829493428101571339539078e-01
            ];
        
    case   14
        
        % DEGREE.             :          14
        % CARDIN.             :         120
        % LEBESGUE CONST.     : 1.72841e+01
        % ABS.DET.VAND.MATRIX : 8.45765e+103
        % CONDITIONING VAND.  : 2.11343e+02
        % POINTS OUT DISK     :           0
        
        pts=[
            9.87930439740756671795907095656730e-01 1.54898180214084663086637760898157e-01
            9.19353038982236303588990722346352e-01 3.93433589966752372824032590870047e-01
            8.30387982764797527579503366723657e-01 5.57185604695428549071323232055875e-01
            7.12583964147506909725393597909715e-01 7.01586839984776666767629649257287e-01
            5.17849624898325311406210857967380e-01 8.55471662881163830149944260483608e-01
            2.91214712227252259246768062439514e-01 9.56657719031419606636745811556466e-01
            1.55518119203507890896398535574008e-02 9.99879063260149303360435624199454e-01
            -2.61321923212860574015081738252775e-01 9.65251704193435644540954854164738e-01
            -4.90994080973322089000987489271211e-01 8.71162908099950561435775853169616e-01
            -6.90420020571746673176960484852316e-01 7.23408733147249849437798729923088e-01
            -8.47314704957777609095614934631158e-01 5.31091132257275000405627451982582e-01
            -9.31144062097659652721404199837707e-01 3.64651526282654958599493966175942e-01
            -9.76389420563607002279127300425898e-01 2.16017821976483392898416013849783e-01
            1.35132208020237482015701857562817e-01 9.59110330659360132976587465236662e-01
            -1.05238642993211228837679982461850e-01 9.62848985066131257681831812078599e-01
            -9.64369192831492472173238184041111e-01 9.02518694513724378802876913141517e-02
            9.60293685676943065132604715472553e-01 0.00000000000000000000000000000000e+00
            9.22910355552496897146852461446542e-01 2.65330809302220527179372311366023e-01
            8.13670960952175725822144158883020e-01 5.10003460825679488799266891874140e-01
            5.71671343392394781801613135030493e-01 7.71593052000173362792168063606368e-01
            3.36224218346981629235159516611020e-01 8.99509442834242167741365392430453e-01
            -3.36224218346981296168252129064058e-01 8.99509442834242278763667854946107e-01
            -5.47398378194864254631113453797298e-01 7.88998718820658195838291248946916e-01
            -7.44090387252807849982616517081624e-01 6.07036620270123394860206644807477e-01
            -9.22910355552496897146852461446542e-01 2.65330809302220749223977236397332e-01
            6.54050938260793635059542339149630e-01 6.05070240803835979193081584526226e-01
            -7.82921552249708385673443444829900e-01 4.25354521745265778953637436643476e-01
            -8.89282958613991758589634173404193e-01 5.53935525578745160957261362000281e-02
            8.65730043763496714959160271973815e-01 1.35738310048240606864311530443956e-01
            7.94523257024214446531118483108003e-01 3.69656856472502515220668328765896e-01
            4.30261393018919913178166325451457e-01 7.63405875774424980129140294593526e-01
            1.75968579343521058788724076293875e-01 8.58457020807286141916847554966807e-01
            -1.22258223485143521780393882636417e-01 8.67736321862669912619026035827119e-01
            -3.81968427637002561070289630151819e-01 7.88678336064846052799737208260922e-01
            -6.05019676063043299052424117689952e-01 6.33927905259003776627935167198302e-01
            -7.59395020804670584091411456029164e-01 2.18321199061157156240398080626619e-01
            6.11896658615709632123014216631418e-01 4.68266095791476322229840434374637e-01
            -6.26163684451212954051868564420147e-01 4.49009685382558476618442000471987e-01
            7.14115570393593857012604075862328e-01 2.29576934607082033812375243542192e-01
            4.65076024826899303565852505926159e-01 5.88532843529876115162835503724637e-01
            2.84358408089210590308226755951182e-01 6.94123124907333854949342821782921e-01
            3.49854731214880310119497153209522e-02 7.49294757390319476897388994984794e-01
            -2.18443379280910426887274411456019e-01 7.17599544892893903558217516547302e-01
            -4.27587506932606831178134143556235e-01 6.16307991751940620339667020743946e-01
            -7.23333606888276858803976665512891e-01 9.04640973571972095701454463778646e-02
            4.98083767592247372846259167999960e-01 3.57166091445821176630914806082728e-01
            -5.70704763626740718684970943286316e-01 2.23497492583962981127498892419680e-01
            5.73907301901681488054407509480370e-01 1.26972089990117942281955265571014e-01
            6.38641314501102869138904338797147e-02 5.84305464227957926937051524873823e-01
            -2.05799163272229274079805350083916e-01 5.50579882677324827788822858565254e-01
            -4.18846345146013043780897078249836e-01 4.12382397745531037447364042236586e-01
            2.75979611538887503741079854080454e-01 4.89666190044305449369232974277111e-01
            3.53562007025745250743398173653986e-01 2.37238092037476200335532894314383e-01
            -4.01098681330276984891014535605791e-01 1.42856056787131285856773388331931e-01
            6.76118690493153484677435471894569e-02 3.91350331799161565982103638816625e-01
            -1.94997263572717516311172403220553e-01 3.45980511351156538868423240273842e-01
            1.64659522627356807999987609036907e-01 1.43086391299526716203160958684748e-01
            -1.07107040330037039943533727637259e-01 1.90038300557361650167109701214940e-01
            -2.15510348399360118776257877470925e-01 -3.37899911183261875136096819005616e-02
            1.88349609398287015649131603822752e-01 -1.10049527061695531782170576207136e-01
            -1.93191872168347773819352397595139e-02 -2.47938357224833455516588287537161e-01
            -2.74198854822091053762278534122743e-01 -2.87300252436209213602325007741456e-01
            3.96955783167381059950429289528984e-01 -1.23512446994978684206012431445743e-02
            -4.15726395779204449532073795126053e-01 -9.19759152065771357076329195479047e-02
            1.36600968987806353771574663369393e-01 -4.03271844290234993124499851546716e-01
            3.46012910834542275484437823251938e-01 -2.48119065493690532653303648658039e-01
            -9.67395451338533135521302597226168e-02 -4.71940740896351296207456016418291e-01
            -5.16483021865079661161246349365683e-01 -2.80600767884982449373865165398456e-01
            -3.35056554709036447103898126442800e-01 -4.82937478312707280103666107606841e-01
            3.19875192859870483008677410907694e-01 -4.93124085606642004186994654446607e-01
            5.80690942770434426201120459154481e-01 -9.10468659367807270355044124698907e-02
            -6.11721444491549970479127296130173e-01 -3.81042093048025373724918551943119e-02
            8.55099353315162402644489247904858e-02 -6.06912767518671181221634469693527e-01
            5.08950651905210604830642751039704e-01 -3.41502987311727457253596185182687e-01
            -5.50625428179119968952193175937282e-01 -5.09390081003461170361390486505115e-01
            -3.68300095261123172640793654863955e-01 -6.53468940817235499629589412506903e-01
            -1.50627608209164737251839483178628e-01 -7.34831913043614615155263436463429e-01
            3.05807992474722456499591771716950e-01 -6.84943857933430932938279056543251e-01
            5.17891700125356790884723068302264e-01 -5.42636898601099204597630887292325e-01
            6.98461468400126461730792470916640e-01 -2.73529146422261892901417468237923e-01
            7.49748227623103447037067326164106e-01 -2.33283509525872151391112652163429e-02
            -7.67161007771208547367791652504820e-01 -7.17958595486246353711834444766282e-02
            -6.88156947293836140921996502584079e-01 -3.46598720690210471140346726315329e-01
            5.98564330859512952476286784531112e-02 -7.68184785524348434648800321156159e-01
            -8.42192885067440166757535280339653e-01 -2.42125054117246912843697259631881e-01
            -5.43318508130850141846224232722307e-01 -6.87545195759497462795195588114439e-01
            -3.06818146286454151994149697202374e-01 -8.20838609349512915436264393065358e-01
            2.28998146953339309916941601841245e-01 -8.45856516308431061190731270471588e-01
            4.76889760656918126890957410068950e-01 -7.35179946455346122746732362429611e-01
            6.61456383830341776430827849253546e-01 -5.74794615301488232006477119284682e-01
            7.94523257024214446531118483108003e-01 -3.69656856472502626242970791281550e-01
            8.69532717558211043495930425706320e-01 -1.08748565900110311099879822904768e-01
            -7.39881110251052520609960083675105e-01 -4.96456008967467665726758241362404e-01
            -4.15568921258815254793361759766412e-02 -8.90036881743152430424004251108272e-01
            -9.52870492917920475228754639829276e-01 -1.19171248534897966653289813621086e-01
            -8.82848918242549962620557835180080e-01 -3.77811792178283756271639504120685e-01
            -6.84289881285495948937125376687618e-01 -6.73729412391420812156184183550067e-01
            -4.71498515663434925038188794133021e-01 -8.36572239844345366677202946448233e-01
            -2.21997162405024511810935905486986e-01 -9.34281126125923844227827430586331e-01
            1.92833764102616156410618941663415e-01 -9.40733278976046416453016263403697e-01
            4.18577054746865073742156937441905e-01 -8.64266863873912050131309570133453e-01
            6.41080675682386025293624243204249e-01 -7.14968202102459149394064752414124e-01
            8.13670960952175725822144158883020e-01 -5.10003460825679377776964429358486e-01
            9.22910355552496786124549998930888e-01 -2.65330809302220693712826005139505e-01
            -1.51772739315891580513806502494845e-02 -9.75798737747193056790706577885430e-01
            -9.99516282291988078512190440960694e-01 -3.10998622698369156513997779711644e-02
            -9.61070940398724538056285382481292e-01 -2.76301732750830253415585957554867e-01
            -8.63421704896663477768470329465345e-01 -5.04482863448639795578287703392562e-01
            -7.54822939153257044253564345126506e-01 -6.55928601699939939351224893471226e-01
            -6.20009547460775078953076899779262e-01 -7.84594265246365774046921615081374e-01
            -4.07683614941688809985009811498458e-01 -9.13123250228618843493677559308708e-01
            -1.70243555722398631768754739823635e-01 -9.85402015288681321081298847275320e-01
            1.08652150085474366592031003619923e-01 -9.94079830940052588950095469044754e-01
            3.79088403840379450926434401480947e-01 -9.25360460617241531444676638784586e-01
            5.95308864276666072079535751981894e-01 -8.03496954638675653370682994136587e-01
            7.74857107102889797189959608658683e-01 -6.32136427974326453593789665319491e-01
            8.78693400099268773217886518978048e-01 -4.77386540051127017392218476743437e-01
            9.42034263469989086026146196672926e-01 -3.35516685797525027190602031623712e-01
            9.95649347969018605475355343514821e-01 -9.31792674840717094486564064936829e-02
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            ];
        
    case   15
        
        % DEGREE.             :          15
        % CARDIN.             :         136
        % LEBESGUE CONST.     : 1.66404e+01
        % ABS.DET.VAND.MATRIX : 6.54684e+121
        % CONDITIONING VAND.  : 2.28466e+02
        % POINTS OUT DISK     :           0
        
        pts=[
            9.82635248222263668615994447463891e-01 1.85547753829493261568117645765597e-01
            9.31144062097659541699101737322053e-01 3.64651526282655180644098891207250e-01
            8.30387982764797527579503366723657e-01 5.57185604695428549071323232055875e-01
            6.90420020571746673176960484852316e-01 7.23408733147249849437798729923088e-01
            5.44204182756027332601433954550885e-01 8.38952804078301195289668612531386e-01
            4.07683614941688809985009811498458e-01 9.13123250228618843493677559308708e-01
            2.00807072855718665715940574045817e-01 9.79630807749081622404219160671346e-01
            -1.55518119203506659242730592040971e-02 9.99879063260149303360435624199454e-01
            -2.31176322114969662324668320252385e-01 9.72911870671437473134801621199585e-01
            -4.35884418475371071544088863447541e-01 9.00002652068530029083603949402459e-01
            -5.95308864276666072079535751981894e-01 8.03496954638675653370682994136587e-01
            -7.34058528759460204859976784064202e-01 6.79086206865885944061744794453261e-01
            -8.47314704957777609095614934631158e-01 5.31091132257275000405627451982582e-01
            -9.42034263469989086026146196672926e-01 3.35516685797525027190602031623712e-01
            -9.92269872363276461690873020415893e-01 1.24098752613259658006406027652702e-01
            9.66709531085391593485667272034334e-01 6.02164639496042750077542393682961e-02
            9.12438524803243833360966164036654e-01 3.24975812141168385327461010092520e-01
            7.87126771091965382609600965224672e-01 5.64433154813099524638175807922380e-01
            6.00530807409489608694741491490277e-01 7.59944793635720872160277394868899e-01
            3.94875484500568585399804533153656e-01 8.84435804206485598477627263491740e-01
            1.35132208020237482015701857562817e-01 9.59110330659360132976587465236662e-01
            -1.35132208020237370993399395047163e-01 9.59110330659360132976587465236662e-01
            -3.94875484500568474377502070638002e-01 8.84435804206485598477627263491740e-01
            -6.23874498438489411000773543491960e-01 7.40900769482698673407128353574080e-01
            -8.04299817309554576638674916466698e-01 5.39680594331266361152188437699806e-01
            -9.64369192831492472173238184041111e-01 9.02518694513724378802876913141517e-02
            -9.04629554901560273272309586900519e-01 3.22194554810618150053613817362930e-01
            8.69969343870674038932122584810713e-01 1.92473288722008195605894798063673e-01
            7.95771306273031053812871959962649e-01 4.00800017787861784235303730383748e-01
            6.54050938260793635059542339149630e-01 6.05070240803835979193081584526226e-01
            4.61407394332906950484840535864350e-01 7.62230832885388953634731024067150e-01
            2.32839538496110609600009411224164e-01 8.60045565920291799066887961089378e-01
            -1.38567658839828765360291384922675e-02 8.90898768764147086507421136047924e-01
            -2.59474208534119565872089197000605e-01 8.52388269072157478412066211603815e-01
            -4.61407394332907061507142998380004e-01 7.62230832885388953634731024067150e-01
            -6.54050938260793746081844801665284e-01 6.05070240803835979193081584526226e-01
            -7.95771306273030942790569497446995e-01 4.00800017787862006279908655415056e-01
            -8.75534417063493219934855460451217e-01 1.65324259210575863487946435270715e-01
            5.08947070130900125484174623125000e-01 6.04415273952933818080168748565484e-01
            2.99538602427975642061852568076574e-01 7.31178206210991055158388007839676e-01
            7.69022759750360274111358194204513e-01 4.79025289395251518964613524076412e-02
            7.25849875152163592773035816207994e-01 2.58520049579236521619662880766555e-01
            6.26163684451213065074171026935801e-01 4.49009685382558310084988306698506e-01
            1.07498416226645562843167169830849e-01 7.62977553930405782978141360217705e-01
            -1.07498416226645479576440322944109e-01 7.62977553930405782978141360217705e-01
            -3.14125624175276785265964463178534e-01 7.03573556587621551372535577684175e-01
            -4.77725566965951309317262030162965e-01 6.04540271578265286933628885890357e-01
            -6.26163684451212954051868564420147e-01 4.49009685382558476618442000471987e-01
            -7.25849875152163481750733353692340e-01 2.58520049579236577130814112024382e-01
            -7.69022759750360274111358194204513e-01 4.79025289395251518964613524076412e-02
            3.79464151399993732116655564823304e-01 5.12168234576706571381521371222334e-01
            6.05509535035596568874893819156568e-01 9.49381872512224112004020071253763e-02
            1.41689598461817556351860503127682e-01 5.96304548117236432602794593549334e-01
            -1.23076071376676920987591756784241e-01 6.00422632045175097559308596828487e-01
            -3.64869002017311094832052731362637e-01 4.92468951086730000188396161320270e-01
            -5.38557382919158889045263549633091e-01 2.92593577716324937387781801589881e-01
            -6.12610579657759113381132465292467e-01 1.90613249528190122072945911213537e-02
            5.07506544647482682464101344521623e-01 2.96527587169388096022260015161010e-01
            2.74615134750987477385564261567197e-01 3.95819567828825991107066784024937e-01
            4.46107067402119139565286332071992e-01 8.42369174866020586911474765656749e-02
            -1.32208712735558425732662612972490e-01 4.34313515942769234889908602781361e-01
            -3.33255598309583844773840155539801e-01 3.08298686421276835201155108734383e-01
            -4.43271520982078559569572462351061e-01 9.80700389517521614601847090852971e-02
            7.24861805490018551445530192722799e-02 4.19563771976409649600014972747886e-01
            2.26836560252544283677877956506563e-01 2.09849178567179883714288735063747e-01
            -7.29064923894545691140933740825858e-02 2.69296640559177791196532325557200e-01
            -2.52953591817856648926721163661568e-01 1.17688222161088609785117853334668e-01
            7.14439861578959611865258239049581e-02 1.02976581209296647045903227990493e-01
            -1.25272607659827034387589606012625e-01 3.89784630168315019901892348741512e-03
            4.36345929424442127708694272314460e-02 -8.33810346206070446273272978032765e-02
            -4.74964379070431375051697386879823e-02 -2.74918398138674635600153806080925e-01
            1.72977149400963564618294299179979e-01 -2.18894821853120008503523763465637e-01
            2.77777312817113819498615612246795e-01 -2.59961868953063474085851680683845e-02
            -2.71531193475778920287666551303118e-01 -1.47520553761654571456318763011950e-01
            -1.58954081640432537403384571916831e-01 -4.25254010896558287679880550058442e-01
            4.93270439150807413208887908240285e-02 -4.51302799229478635023582455687574e-01
            2.81478444295007457220947344467277e-01 -3.56198342571980242077245293330634e-01
            4.22730558081228058320277796156006e-01 -1.65548328647850989669976229379245e-01
            -4.62999456609476334190134139134898e-01 -1.33109374913382760841074059499078e-01
            -3.62734749950658741379783123193192e-01 -3.57136758297679934859303330085822e-01
            -2.67156634660980829565346539311577e-01 -5.51617973759187751880972427898087e-01
            2.49872163256505047668198926658079e-01 -5.59659680919652458008783924015006e-01
            4.62636303666102643994406662386609e-01 -4.02023266674627066663560981396586e-01
            5.89047158431277750878507504239678e-01 -1.69347280939523642251387514079397e-01
            -5.60100252857117619775806360848946e-01 -3.04297633011712032047313414295786e-01
            -9.91309800209123383185616518176175e-03 -6.37346901769467066856122983153909e-01
            -7.52320978650438565438207660918124e-01 -1.66444592508463440339028238668106e-01
            -5.65601817381676474916218921862310e-01 -5.23244915376544161844662994553801e-01
            -3.99009993750634672782240386368358e-01 -6.59152245069362208162999650085112e-01
            -2.01352002463147106148255716107087e-01 -7.43739220692940850021557253057836e-01
            2.01352002463146967370377637962520e-01 -7.43739220692940850021557253057836e-01
            3.78317441514472774244381980679464e-01 -6.71242557306079734580350759642897e-01
            5.65601817381676585938521384377964e-01 -5.23244915376544161844662994553801e-01
            6.88156947293836140921996502584079e-01 -3.46598720690210637673800420088810e-01
            7.57133471573528837872402164066443e-01 -1.42967001492926820693796230443695e-01
            -6.94303994429875226401804866327439e-01 -3.77209367462086275768484711079509e-01
            1.22883421403890934209268692711703e-02 -7.90059453604517236691151538252598e-01
            -8.89282958613991647567331710888539e-01 -5.53935525578745091568322322927997e-02
            -8.39358674760744216669650086259935e-01 -2.98947556019653393200030677689938e-01
            -7.24083503154279495284129097854020e-01 -5.19226065029542827033992580254562e-01
            -5.52432551846628205893807717075106e-01 -6.99078609175290854871320789243327e-01
            -3.63248760717743091053222315167659e-01 -8.13598773341786873913861199980602e-01
            -1.24309077223386965638241008491605e-01 -8.82292400208743998746285797096789e-01
            1.24309077223386868493726353790407e-01 -8.82292400208743998746285797096789e-01
            3.63248760717742980030919852652005e-01 -8.13598773341786873913861199980602e-01
            5.73906578899958330630681757611455e-01 -6.81559876196935698899892486224417e-01
            7.24083503154279273239524172822712e-01 -5.19226065029543049078597505285870e-01
            8.39358674760744105647347623744281e-01 -2.98947556019653504222333140205592e-01
            8.89282958613991647567331710888539e-01 -5.53935525578745091568322322927997e-02
            8.57652710627436531964917776349466e-01 -4.31967349118446530820136786132935e-01
            -9.51763954959537139544067940732930e-01 -1.79718429944487612104353502218146e-01
            -8.65056165629651041548697776306653e-01 -4.35696190398834337642597347439732e-01
            -7.10996730239269902895671293663327e-01 -6.57751464925011242179664350260282e-01
            -5.01580426673295720618739323981572e-01 -8.28595447489404168806004236103036e-01
            -2.53112014457726020033589975355426e-01 -9.34926546932476121298805082915351e-01
            1.50632231510911766131366462673213e-02 -9.68466023838850054161753178050276e-01
            2.82065666536236270278692472857074e-01 -9.26602557617558164615445548406569e-01
            5.27107007633255908451985760621028e-01 -8.12595559011890067324657138669863e-01
            7.31108788497465744704584267310565e-01 -6.35321398509210655269896506069927e-01
            9.45714351462050739627329676295631e-01 -2.09231224870104121071889835548063e-01
            -9.98065597133594328838057663233485e-01 -6.21696374314805247363047158160043e-02
            -9.61070940398724538056285382481292e-01 -2.76301732750830253415585957554867e-01
            -8.93115016186791565999669728626031e-01 -4.49828376007635810651663632597774e-01
            -7.74857107102889908212262071174337e-01 -6.32136427974326342571487202803837e-01
            -6.20009547460775078953076899779262e-01 -7.84594265246365774046921615081374e-01
            -4.63663531985327337370250688763917e-01 -8.86011359468314774190389471186791e-01
            -3.20825769815367811688133770076092e-01 -9.47138229311000112708995857246919e-01
            -1.08652150085474491492121273950033e-01 -9.94079830940052588950095469044754e-01
            1.08652150085474366592031003619923e-01 -9.94079830940052588950095469044754e-01
            3.20825769815367700665831307560438e-01 -9.47138229311000112708995857246919e-01
            5.17849624898325422428513320483034e-01 -8.55471662881163830149944260483608e-01
            6.90420020571746673176960484852316e-01 -7.23408733147249849437798729923088e-01
            7.94141650844753299587353012611857e-01 -6.07732703080531955208698491333053e-01
            8.93115016186791454977367266110377e-01 -4.49828376007636032696268557629082e-01
            9.69198999199666011428178080677753e-01 -2.46278906832001820426114591100486e-01
            9.99516282291988078512190440960694e-01 -3.10998622698367525873930361512976e-02
            ];
        
    case   16
        
        % DEGREE.             :          16
        % CARDIN.             :         153
        % LEBESGUE CONST.     : 2.40925e+01
        % ABS.DET.VAND.MATRIX : 2.59044e+140
        % CONDITIONING VAND.  : 2.76931e+02
        % POINTS OUT DISK     :           0
        
        pts=[
            9.98065597133594328838057663233485e-01 6.21696374314805247363047158160043e-02
            9.61070940398724538056285382481292e-01 2.76301732750830253415585957554867e-01
            8.78693400099268773217886518978048e-01 4.77386540051127017392218476743437e-01
            7.54822939153257044253564345126506e-01 6.55928601699939939351224893471226e-01
            5.95308864276666072079535751981894e-01 8.03496954638675653370682994136587e-01
            4.35884418475371182566391325963195e-01 9.00002652068530029083603949402459e-01
            2.31176322114969773346970782768039e-01 9.72911870671437362112499158683931e-01
            1.55518119203507890896398535574008e-02 9.99879063260149303360435624199454e-01
            -2.00807072855718332649033186498855e-01 9.79630807749081733426521623187000e-01
            -3.50126449191390465287554434326012e-01 9.36702444523675126220041420310736e-01
            -4.90994080973322089000987489271211e-01 8.71162908099950561435775853169616e-01
            -6.67588140216153647088503930717707e-01 7.44530775081015172567333593178773e-01
            -8.12657913828250144661069498397410e-01 5.82741036046301008610726057668217e-01
            -9.19353038982236192566688259830698e-01 3.93433589966752705890939978417009e-01
            -9.69198999199666011428178080677753e-01 2.46278906832001820426114591100486e-01
            -9.95649347969018605475355343514821e-01 9.31792674840717094486564064936829e-02
            9.51763954959537250566370403248584e-01 1.79718429944487639859929117847059e-01
            8.90469872690628050371230983728310e-01 3.81073150264182702873227981399396e-01
            8.04299817309554798683279841498006e-01 5.39680594331266028085281050152844e-01
            6.68729206031876910465427954477491e-01 7.00681517539821552276180227636360e-01
            5.01580426673295831641041786497226e-01 8.28595447489404279828306698618690e-01
            3.10746438299295490992335544433445e-01 9.17382140171822735297268991416786e-01
            7.52432663758195036107778719269845e-02 9.65656145264466858613161548419157e-01
            -1.94498349403578990601815235095273e-01 9.48853904508599854672468154603848e-01
            -4.22190307933589970001975189006771e-01 8.71727413764688319197659893688979e-01
            -6.23874498438489411000773543491960e-01 7.40900769482698673407128353574080e-01
            -8.04299817309554576638674916466698e-01 5.39680594331266361152188437699806e-01
            -9.12438524803243722338663701521000e-01 3.24975812141168496349763472608174e-01
            -9.64369192831492472173238184041111e-01 9.02518694513724378802876913141517e-02
            8.42524337129061606788127392064780e-01 3.29946565703570171557146295526763e-01
            3.94400013584935094090866414262564e-01 8.14346746882768535158447775756940e-01
            1.81695671846369172230950539415062e-01 8.86396456280507449854155765933683e-01
            -4.22014869601116324182576988732762e-02 9.03842369759628172509735577477841e-01
            -2.90291835657180807839239378154161e-01 8.56996292305357232343965279142139e-01
            -4.92410666622815074155994352622656e-01 7.59107192872271352435120661539258e-01
            -6.64196014914965271458413553773426e-01 6.14455570928789218676513428363251e-01
            -7.95065559233166241348556013690541e-01 4.31952255921412675565562722113100e-01
            -8.76957473698849909737873531412333e-01 2.22839817353353647488845012958336e-01
            8.80252467253297421478919204673730e-01 1.38015289155655002106826145791274e-01
            7.24083503154279606306431560369674e-01 5.19226065029542827033992580254562e-01
            5.73906578899958441652984220127109e-01 6.81559876196935698899892486224417e-01
            7.60379039442491633238319082011003e-01 3.25401279528604037150074645978748e-01
            2.35597651203856894852606274071150e-01 7.73952352496392048664120011380874e-01
            -1.37729929762238745283298158028629e-01 7.97206976659864952061695930751739e-01
            -3.52637902129816493967950918886345e-01 7.28117440505963720021043172891950e-01
            -6.57454062900346025699604979308788e-01 4.71447401481121353139513985297526e-01
            -8.02763189748150640667745392420329e-01 1.00397999844859489582127309859061e-01
            7.88626534254846078830780697899172e-01 4.91236506340636813749966904651956e-02
            5.80020025876412326226727600442246e-01 5.36583370190274799860219445690745e-01
            4.09181476770282881449958267694456e-01 6.75955222370918429319885945005808e-01
            3.68531382437769922111314713220054e-02 7.89295122108335633903664074750850e-01
            -5.08947070130900014461872160609346e-01 6.04415273952933929102471211081138e-01
            -7.35748147910326943943459809815977e-01 2.88131231262685638672138566107606e-01
            6.57898330842110135030509354692185e-01 1.89141551517662848125311825242534e-01
            5.80026829089667295136223401641473e-01 3.63556897571108039723952742861002e-01
            2.19620352234453697359839452474262e-01 6.48360733789267840165848610922694e-01
            -1.58251082263040798725484137321473e-01 6.66004005391799158708465711242752e-01
            4.25957857670647532799534928926732e-01 5.05858959302373456523582717636600e-01
            -3.59888683215321858455126857734285e-01 5.54809443783529698812628794257762e-01
            -5.37420320863787659426691334374482e-01 3.85373561548416787569948382952134e-01
            -6.45697908995198632808865113474894e-01 1.42855148794420794278181574554765e-01
            2.62157881733017343595104620135317e-02 5.61471687717300382836072003556183e-01
            5.14967561741298762711949166259728e-01 1.48049871907020624828277277629240e-01
            2.33558550932941838551570867821283e-01 4.82245536530477414416395731677767e-01
            -1.87607133107599216570449129903864e-01 5.01910268698112171925629354518605e-01
            3.94434358803892104461397138948087e-01 3.21783622243433653498101421064348e-01
            -3.62734749950658685868631891935365e-01 3.57136758297679879348152098827995e-01
            -4.67988772537923702898865485622082e-01 2.00273991640426834992538829283149e-01
            3.38436426234635157062058397059445e-01 1.44832564317637774164992947589781e-01
            1.90633161523624333755222437503107e-01 3.14920123232546145253252234397223e-01
            -5.72500380661533463111378949861319e-03 3.68080032901417497637197584481328e-01
            -2.33871441864184820280669896419568e-01 2.45045969753580478167620526619430e-01
            -3.37264189424605898004472237516893e-01 3.15633512775341656952932112290000e-02
            -2.91411896182525295348475502521524e-03 1.87358653300429833787887901053182e-01
            -1.35068912509633681784393388625176e-01 7.89185068655518379143742890846625e-02
            3.04432762626238688496993489707165e-02 7.73580740852482413860702337160546e-03
            -1.48807306502567432415773396314762e-01 -1.13877752819965857855599722370243e-01
            3.19004612710110935225316097785253e-02 -1.84645925020215401213263817226107e-01
            2.05498407739947791528223319801327e-01 -7.31906973824973522413372961636924e-02
            -3.11917346745731571822091154899681e-01 -1.95507685497008393271656245815393e-01
            -8.51016801698622205041289134896942e-02 -3.58152747192535925169210031526745e-01
            1.50078348387281568232509698646027e-01 -3.36143088036389370287793099123519e-01
            3.22745376284550911094584080274217e-01 -2.31434373252115133112738476484083e-01
            4.22487963296200974561145358165959e-01 -5.28386789717829111201474745485029e-02
            -5.09041415750371317727740461123176e-01 -0.00000000000000000000000000000000e+00
            -3.15610537818180048041938334790757e-01 -3.99390975570632522639868966507493e-01
            8.66610206273069755278370962514600e-02 -5.01610436945819415655023476574570e-01
            -4.85819473983981064968418195348931e-01 -2.26030513239677210979294841308729e-01
            -1.23870467753896457141493669951160e-01 -5.21312249458896648235395332449116e-01
            3.18982440767952624138814599064062e-01 -4.30535197979426020875592939773924e-01
            5.16753061589756867988398880697787e-01 -2.21142481209002511599592821767146e-01
            -6.58434727513669271203866628638934e-01 -6.16205551893828570864108939986181e-02
            -4.99173365880521691106963544370956e-01 -4.33773367109321850154657340681297e-01
            -3.06626195219769470678983225297998e-01 -5.85929824827935763487118947523413e-01
            1.02845977502097028766225861318162e-02 -6.61231888422635027957596776104765e-01
            2.50695659466247111613057541035232e-01 -6.11951852307541566133863852883223e-01
            6.58434727513669271203866628638934e-01 -6.16205551893829472920316447925870e-02
            4.56995529337283656889923122434993e-01 -5.09666387356552452736480063322233e-01
            -6.50080768168799560946524707105709e-01 -2.78199559412058206042672736657551e-01
            6.21330061794057031399063362187007e-01 -3.37563259717335251153258468548302e-01
            -7.80618188839678817103617802786175e-01 -1.22393573504031988319340484849818e-01
            -5.08947070130900125484174623125000e-01 -6.04415273952933818080168748565484e-01
            -1.34518798878713430333675660222070e-01 -7.78620341585458231037364384974353e-01
            7.65817447207118862095853728533257e-01 -1.94598512675711904362785276134673e-01
            -6.57454062900346025699604979308788e-01 -4.71447401481121131094909060266218e-01
            -3.52637902129816549479102150144172e-01 -7.28117440505963608998740710376296e-01
            6.28475526452341309857629880752938e-02 -8.06572180473625111041258151090005e-01
            2.59553500014057469957862167575513e-01 -7.66250923534794958769111872243229e-01
            4.40270432259555588583310736794374e-01 -6.78727075977861415978509285196196e-01
            6.10664585519031577476312122598756e-01 -5.30657385871847520064648051629774e-01
            7.38678080481005916446690662269248e-01 -3.72044311553389073221609351094230e-01
            8.87130064844290022740835865988629e-01 -8.30233352474009367316654106616625e-02
            -9.04827052466019354781678885046858e-01 -0.00000000000000000000000000000000e+00
            -8.76957473698849798715571068896679e-01 -2.22839817353353175644059547266806e-01
            -7.95065559233166019303951088659232e-01 -4.31952255921412564543260259597446e-01
            -6.64196014914964938391506166226463e-01 -6.14455570928789107654210965847597e-01
            -4.92410666622815129667145583880483e-01 -7.59107192872271130390515736507950e-01
            -2.90291835657180863350390609411988e-01 -8.56996292305357010299360354110831e-01
            -7.02904465667318761878945565513277e-02 -9.02092704768082009891827510728035e-01
            1.54040974725632384689788523246534e-01 -8.91618400987732817597475332149770e-01
            6.04051809173053899648664355481742e-01 -6.73671586686795720311238255817443e-01
            7.51357510848275178538813179329736e-01 -5.04156608373061532191172773309518e-01
            8.52378085937547713690776163275586e-01 -3.03584573863342088717587330393144e-01
            4.00034941319343861909629822548595e-01 -8.25981597063743855358097789576277e-01
            -9.61095889666325531308643803640734e-01 -1.20199962098270676791322841836518e-01
            -9.01890459132705379907690712570911e-01 -3.53195328037234190787785337306559e-01
            -7.87126771091965271587298502709018e-01 -5.64433154813099524638175807922380e-01
            -6.23874498438489411000773543491960e-01 -7.40900769482698451362523428542772e-01
            -4.22190307933590081024277651522425e-01 -8.71727413764688208175357431173325e-01
            -2.23913492852208073635722485050792e-01 -9.42346055194510534569474202726269e-01
            1.50632231510911766131366462673213e-02 -9.68466023838850054161753178050276e-01
            2.53112014457725853500136281581945e-01 -9.34926546932476121298805082915351e-01
            6.00530807409489386650136566458968e-01 -7.59944793635720872160277394868899e-01
            7.50513546220703209321811755216913e-01 -6.12276699671934210300605627708137e-01
            8.78187814156807355914224899606779e-01 -4.08582308829469775801612740906421e-01
            9.38749830407518026653690412786091e-01 -2.38541602098643917839737582653470e-01
            9.68114640301910855946232459245948e-01 -3.01228029079835223080774397885762e-02
            4.28163707059645515240475788232288e-01 -8.84061130748925005917726593906991e-01
            -9.92269872363276461690873020415893e-01 -1.24098752613259338817286447920196e-01
            -9.42034263469989197048448659188580e-01 -3.35516685797524916168299569108058e-01
            -8.78693400099268773217886518978048e-01 -4.77386540051127017392218476743437e-01
            -7.74857107102889908212262071174337e-01 -6.32136427974326342571487202803837e-01
            -6.20009547460775078953076899779262e-01 -7.84594265246365774046921615081374e-01
            -4.63663531985327337370250688763917e-01 -8.86011359468314774190389471186791e-01
            -3.20825769815367811688133770076092e-01 -9.47138229311000112708995857246919e-01
            -1.08652150085474491492121273950033e-01 -9.94079830940052588950095469044754e-01
            1.08652150085474366592031003619923e-01 -9.94079830940052588950095469044754e-01
            3.20825769815367700665831307560438e-01 -9.47138229311000112708995857246919e-01
            5.44204182756027110556829029519577e-01 -8.38952804078301306311971075047040e-01
            7.34058528759460204859976784064202e-01 -6.79086206865885944061744794453261e-01
            8.63421704896663477768470329465345e-01 -5.04482863448639795578287703392562e-01
            9.42034263469989086026146196672926e-01 -3.35516685797525027190602031623712e-01
            9.92269872363276461690873020415893e-01 -1.24098752613259658006406027652702e-01
            ];
        
    case   17
        
        % DEGREE.             :          17
        % CARDIN.             :         171
        % LEBESGUE CONST.     : 2.88638e+01
        % ABS.DET.VAND.MATRIX : 1.07727e+160
        % CONDITIONING VAND.  : 4.04172e+02
        % POINTS OUT DISK     :           0
        
        pts=[
            9.87930439740756671795907095656730e-01 1.54898180214084663086637760898157e-01
            9.52013107532729940807314505946124e-01 3.06057254587887295205206328319036e-01
            8.78693400099268773217886518978048e-01 4.77386540051127017392218476743437e-01
            7.54822939153257044253564345126506e-01 6.55928601699939939351224893471226e-01
            5.95308864276666072079535751981894e-01 8.03496954638675653370682994136587e-01
            4.07683614941688809985009811498458e-01 9.13123250228618843493677559308708e-01
            2.00807072855718665715940574045817e-01 9.79630807749081622404219160671346e-01
            1.55518119203507890896398535574008e-02 9.99879063260149303360435624199454e-01
            -1.70243555722398520746452277307981e-01 9.85402015288681321081298847275320e-01
            -3.50126449191390465287554434326012e-01 9.36702444523675126220041420310736e-01
            -5.17849624898325422428513320483034e-01 8.55471662881163830149944260483608e-01
            -6.67588140216153647088503930717707e-01 7.44530775081015172567333593178773e-01
            -7.74857107102889797189959608658683e-01 6.32136427974326453593789665319491e-01
            -8.93115016186791454977367266110377e-01 4.49828376007636032696268557629082e-01
            -9.69198999199666011428178080677753e-01 2.46278906832001820426114591100486e-01
            -9.99516282291988078512190440960694e-01 3.10998622698367525873930361512976e-02
            9.68372800806142963736533602059353e-01 1.21110052810969731629242573944794e-01
            8.97212040882091055316038819000823e-01 3.83958435158289868027736702060793e-01
            7.93086479827163737255091291444842e-01 5.68706744947136799694931141857523e-01
            6.51510456108459146129519012902165e-01 7.26600063180810162855038925044937e-01
            5.05378129101968553982260345946997e-01 8.34869135169341136659681978926528e-01
            3.13099246424719701575867247811402e-01 9.24328073857590082340607295918744e-01
            1.06035454489099068009494430953055e-01 9.70139169719633653876655898784520e-01
            -7.58129687012309844140744985452329e-02 9.72967584254334716398204818688100e-01
            -2.55028445125500879964164369084756e-01 9.42005317612315429265379407297587e-01
            -5.56304035636626781347047199233202e-01 8.01834986869126797870421796687879e-01
            -7.16380022460453291799353792157490e-01 6.62731612081821852555663099337835e-01
            -9.19347028040984670660407118703006e-01 3.27436357579940773288029731702409e-01
            -9.68372800806142963736533602059353e-01 1.21110052810970036940574345862842e-01
            -8.36295724315882393895549284934532e-01 4.88633606614307047788514637431945e-01
            5.53504183845284658538332678290317e-01 7.47072574905859654315065654373029e-01
            -4.05275082863474522820723677796195e-01 8.36801303130384477313441493606661e-01
            9.17754625683981140049638725031400e-01 0.00000000000000000000000000000000e+00
            8.89486864723778691832478671130957e-01 2.26023605953463463391628351928375e-01
            8.06424932499089086590515762509312e-01 4.38123705371192939672653210436692e-01
            6.73685610291772207780525150155881e-01 6.23234507589355724022084359603468e-01
            4.00034941319344083954234747579903e-01 8.25981597063744077402702714607585e-01
            1.84291619983396021131838438122941e-01 8.99060705274254523544641415355727e-01
            -1.42727473276691042142205745335559e-02 9.17643635431568238658428526832722e-01
            -2.12163138969623438212508403921674e-01 8.92894369691566991420472731988411e-01
            -5.91135309559814858815229854371864e-01 7.02020369188790849257486570422770e-01
            -9.06677730926089053831162800634047e-01 1.42158521401507143000486621531309e-01
            -7.18560849174357030300086535135051e-01 5.49892990415564386630364879238186e-01
            -8.31855500437974404448482346197125e-01 3.55989355550741404243808574392460e-01
            -4.68419411369447369342111642254167e-01 7.22121937142999437142520946508739e-01
            5.02635898407678083366079135885229e-01 6.78414916857259742144492520310450e-01
            8.20687135869564876600179559318349e-01 1.02639667578131310587963298530667e-01
            7.70131165612080503812819642917020e-01 3.01596193767953935083170335929026e-01
            6.72133574057836691295619857555721e-01 4.81973790746527575024060752184596e-01
            2.89582784665928683054403336427640e-01 7.74728395741027031640157929359702e-01
            8.98640826888601196298367312920163e-02 8.22184117448658047244691715604858e-01
            -1.15390426653849978433363787644339e-01 8.18991651837173151307069929316640e-01
            -3.37187198368301244144618067366537e-01 7.55226502182540504470864561881172e-01
            -5.71032987105135059202609681960894e-01 5.98317310446660699518872661428759e-01
            -8.01605664844390330969758906576317e-01 2.03692499694323375658555619338586e-01
            -6.98522832573591490046283070114441e-01 4.08135209900885587153140932059614e-01
            5.03294534740448940013379797164816e-01 5.27342271267012940860752223670715e-01
            3.57919281279160061970401329745073e-01 6.35050429378066216834497481613653e-01
            -4.15535632856028791515257125865901e-01 5.98936889489699475497275216184789e-01
            6.90411580359330478984247747575864e-01 1.52747666776719914771831554389792e-01
            6.21330061794057142421365824702661e-01 3.37563259717335306664409699806129e-01
            1.84782503976524098421307940043334e-01 6.82536025587049843821318972914014e-01
            -1.09967916686177433205351405831607e-02 7.07021265997704762007458612060873e-01
            -2.05919897797178841347687239249353e-01 6.76459160401571835130596355156740e-01
            -5.47906714883044321240390672755893e-01 4.46987954855687819044618436237215e-01
            -6.98572313281296985998380932869622e-01 1.09529553622835201798224602498522e-01
            -6.20660105160601216489624221139820e-01 2.88765950377420421713026144061587e-01
            5.87785252292473248125759255344747e-01 0.00000000000000000000000000000000e+00
            4.55449580189088099047012292430736e-01 3.71560469800152226582667935872450e-01
            3.04384372420424764715818355398369e-01 5.02833627195666488240988201141590e-01
            -3.04384372420424820226969586656196e-01 5.02833627195666488240988201141590e-01
            5.23380599690806636026252363080857e-01 2.04964561631889835568998137205199e-01
            1.12870317807341247196006861486239e-01 5.50634193467614996286840778338956e-01
            -7.84192529757851558569470284965064e-02 5.56586151840288700931580478936667e-01
            -5.59637948662697448654057552630547e-01 5.23745174132342339312629064806970e-02
            -4.35443885380908835003310741740279e-01 3.12248261647429359300076612271369e-01
            2.31710871400610640691297703597229e-01 3.57208730576990263205772180299391e-01
            -2.20489646426447610805254839760892e-01 3.64242118575536621971622253113310e-01
            -4.05347466485949048387027460194076e-01 1.30312841036781895676099907177559e-01
            3.48971230364127860124057178836665e-01 1.89593057398741271502018435057835e-01
            -6.17636929971610449535690534617061e-03 3.97099860863648679298876231769100e-01
            2.45688309585854108840763387888728e-01 3.85216109594820471651033244597784e-02
            1.15308431451889070729066588683054e-01 2.20342065012954752001306246711465e-01
            -1.15308431451889098484642204311967e-01 2.20342065012954752001306246711465e-01
            -2.42818174828921318209040691726841e-01 5.37214477729293302110669117155339e-02
            -1.31295532300734242581174981978620e-02 9.31879255463620714827754909492796e-02
            -6.90810100186229764096523808802885e-02 -6.39076575260162649261275191747700e-02
            8.76283971405607681193217217696656e-02 -3.43167400874825750944907554185193e-02
            3.86757835168105275694405698061473e-03 -2.48659811420667342662937926434097e-01
            1.92699126534303516455892690828477e-01 -1.57205936945729651821324068805552e-01
            -2.21558457520997120759886911400827e-01 -1.69552019008647969489089746275567e-01
            -1.50554159949312538202548239496537e-01 -3.67504955010966527506610646014451e-01
            -4.15726395779204449532073795126053e-01 -9.19759152065771357076329195479047e-02
            5.94027421780045430943495432529744e-02 -4.21615131781804930444224055463565e-01
            2.53470186494126892284839414060116e-01 -3.42112364120748690954343373960000e-01
            3.67627081832808677575030742445961e-01 -2.14798356205881169067595237720525e-01
            4.25573334582037943185639505827567e-01 -1.32416773250224259483864841513423e-02
            -3.43289942834026529094870738845202e-01 -3.37992037864078498365927316626767e-01
            -5.23380599690806636026252363080857e-01 -2.04964561631889835568998137205199e-01
            -2.29152183381391816308081388342544e-01 -5.13251400883818487130838548182510e-01
            5.23380599690806636026252363080857e-01 -2.04964561631889724546695674689545e-01
            9.14112569320840728792632035037968e-03 -5.87714167460328562597737800388131e-01
            2.22822573092468334543525543267606e-01 -5.43913231805384311812190389900934e-01
            4.18846345146012877247443384476355e-01 -4.12382397745530870913910348463105e-01
            -4.99173365880521691106963544370956e-01 -4.33773367109321850154657340681297e-01
            -7.01640755515182679147301314515062e-01 -8.77510695096274523363533148767601e-02
            -6.50080768168799560946524707105709e-01 -2.78199559412058206042672736657551e-01
            -3.66174981400515797691497255073045e-01 -6.04909813936203022421977948397398e-01
            -1.63466345017260733518682513931708e-01 -6.87952581248662653479186701588333e-01
            6.21330061794057031399063362187007e-01 -3.37563259717335251153258468548302e-01
            6.94828047450888997005336022994015e-01 -1.31202074966766973895815340256377e-01
            3.39993813631146724052456420395174e-02 -7.28175323553045039659536996623501e-01
            2.12286389057221136500430702653830e-01 -6.97373464354432148759599385812180e-01
            3.77496130272824970131040345222573e-01 -6.23612003888394328399158439424355e-01
            5.19451354267116305329921033262508e-01 -5.11434795760628002625480803544633e-01
            -6.26872567858446694266660870198393e-01 -5.11409112994704928567557544738520e-01
            -8.27080574274561719505527435103431e-01 -0.00000000000000000000000000000000e+00
            -8.07552722675354606174380478478270e-01 -1.78664144253850021293317240633769e-01
            -7.49891295660543910628348385216668e-01 -3.48891560567018588212562235639780e-01
            -4.92368397336682084297621031510062e-01 -6.64556722670417321374713992554462e-01
            -3.13536654749127896568694495726959e-01 -7.65347661178281124527700285398168e-01
            -1.15390426653850061700090634531080e-01 -8.18991651837173040284767466800986e-01
            6.72133574057836469251014932524413e-01 -4.81973790746527686046363214700250e-01
            7.70131165612080503812819642917020e-01 -3.01596193767953713038565410897718e-01
            8.20687135869564765577877096802695e-01 -1.02639667578131560388143839190889e-01
            1.17796696706232981166628803748608e-01 -8.36070322417699296480009252263699e-01
            3.44218660864878234217911767700571e-01 -7.70975459593186918105800486955559e-01
            5.23491374998999114964703949226532e-01 -6.62454848336241908235422215511790e-01
            -7.51357510848275400583418104361044e-01 -5.04156608373061199124265385762556e-01
            -8.73714433149944014189713925588876e-01 -2.80885461122173329950868492232985e-01
            -6.12682103735141092570870569034014e-01 -6.83296562794681316965750283998204e-01
            -4.50612088996721560363312164554372e-01 -7.99513788633038346986836586438585e-01
            -2.39829403821237019389656097700936e-01 -8.85864216472871168761571425420698e-01
            -1.42727473276692135017995610724029e-02 -9.17643635431568016613823601801414e-01
            2.39829403821236852856202403927455e-01 -8.85864216472871168761571425420698e-01
            6.73685610291772096758222687640227e-01 -6.23234507589355612999781897087814e-01
            8.06424932499088864545910837478004e-01 -4.38123705371192828650350747921038e-01
            8.89486864723778358765571283583995e-01 -2.26023605953463851969686970733164e-01
            -9.27977923588832576129448170831893e-01 -5.78038670199886514544651561209321e-02
            4.81484404456514136771971834605210e-01 -7.95397436490627329774838472076226e-01
            -8.65056165629651041548697776306653e-01 -4.35696190398834337642597347439732e-01
            -7.50513546220703320344114217732567e-01 -6.12276699671934099278303165192483e-01
            -9.45857548973212725051951110799564e-01 -2.40347713289301184724777726842149e-01
            -5.80971899178317152667716527503217e-01 -7.84146146198620885847674344404368e-01
            -3.69958727564431211209949879048509e-01 -9.03074784351726056463860459189164e-01
            -1.36155357822940276113499180610233e-01 -9.66372208193673531972933687939076e-01
            1.36155357822940165091196718094579e-01 -9.66372208193673531972933687939076e-01
            3.69958727564431322232252341564163e-01 -9.03074784351726056463860459189164e-01
            5.05378129101968665004562808462651e-01 -8.34869135169341025637379516410874e-01
            6.28598147073706425835837308113696e-01 -7.46510479315938724198531417641789e-01
            7.93086479827163404188183903897880e-01 -5.68706744947136910717233604373178e-01
            9.08719097980839829453714173723711e-01 -3.55869536765790639432083253268502e-01
            9.68372800806142852714231139543699e-01 -1.21110052810970023062786538048385e-01
            -9.80387111455245374713740602601320e-01 -6.10684422313683833438702208695759e-02
            -9.82635248222263668615994447463891e-01 -1.85547753829493261568117645765597e-01
            -9.19353038982236303588990722346352e-01 -3.93433589966752372824032590870047e-01
            -8.47314704957777498073312472115504e-01 -5.31091132257275000405627451982582e-01
            -7.74857107102889908212262071174337e-01 -6.32136427974326342571487202803837e-01
            -6.20009547460775078953076899779262e-01 -7.84594265246365774046921615081374e-01
            -4.63663531985327337370250688763917e-01 -8.86011359468314774190389471186791e-01
            -2.91214712227252259246768062439514e-01 -9.56657719031419606636745811556466e-01
            -7.76838472890062359255125556956045e-02 -9.96978043825629267615795470192097e-01
            7.76838472890061110254222853654937e-02 -9.96978043825629267615795470192097e-01
            2.00807072855718332649033186498855e-01 -9.79630807749081733426521623187000e-01
            4.07683614941688698962707348982804e-01 -9.13123250228618843493677559308708e-01
            5.70032258213782561107052515581017e-01 -8.21622312620400530036590680538211e-01
            7.12583964147506909725393597909715e-01 -7.01586839984776666767629649257287e-01
            8.47314704957777609095614934631158e-01 -5.31091132257275000405627451982582e-01
            9.19353038982236192566688259830698e-01 -3.93433589966752705890939978417009e-01
            9.69198999199666011428178080677753e-01 -2.46278906832001820426114591100486e-01
            9.98065597133594328838057663233485e-01 -6.21696374314805247363047158160043e-02
            ];
        
    case   18
        
        % DEGREE.             :          18
        % CARDIN.             :         190
        % LEBESGUE CONST.     : 3.23549e+01
        % ABS.DET.VAND.MATRIX : 9.95608e+181
        % CONDITIONING VAND.  : 3.82217e+02
        % POINTS OUT DISK     :           0
        
        pts=[
            9.95649347969018605475355343514821e-01 9.31792674840715706707783283491153e-02
            9.61070940398724538056285382481292e-01 2.76301732750830253415585957554867e-01
            8.93115016186791565999669728626031e-01 4.49828376007635810651663632597774e-01
            7.94141650844753521631957937643165e-01 6.07732703080531622141791103786090e-01
            6.67588140216153647088503930717707e-01 7.44530775081015172567333593178773e-01
            4.90994080973322200023289951786865e-01 8.71162908099950450413473390653962e-01
            2.91214712227252259246768062439514e-01 9.56657719031419606636745811556466e-01
            1.08652150085474491492121273950033e-01 9.94079830940052588950095469044754e-01
            -7.76838472890061110254222853654937e-02 9.96978043825629267615795470192097e-01
            -2.31176322114969662324668320252385e-01 9.72911870671437473134801621199585e-01
            -4.07683614941688698962707348982804e-01 9.13123250228618843493677559308708e-01
            -5.70032258213782561107052515581017e-01 8.21622312620400530036590680538211e-01
            -6.90420020571746673176960484852316e-01 7.23408733147249849437798729923088e-01
            -8.12657913828250144661069498397410e-01 5.82741036046301008610726057668217e-01
            -9.06672601177072490230557377799414e-01 4.21835032062062353919174029215355e-01
            -9.76389420563607002279127300425898e-01 2.16017821976483392898416013849783e-01
            9.03068769110925928522703998169163e-01 3.86464799432759409381077375655877e-01
            5.59935419747482732155674511886900e-01 8.07069122601240418468648840644164e-01
            -9.74694044904629741665758047020063e-01 1.21900622523338497682487968631904e-01
            9.75444693719451971780642907106085e-01 3.03508768831202667415336549083804e-02
            9.45857548973212836074253573315218e-01 2.40347713289301212480353342471062e-01
            8.26908623205479309525856024265522e-01 5.18300738086902912371556340076495e-01
            6.95422634900311420480534252419602e-01 6.84690357096781387902240112453001e-01
            4.25386910258038208887398923252476e-01 8.78327673943004927359368139150320e-01
            2.25608847715350141482915091728501e-01 9.49481002477438607023429995024344e-01
            4.55171387624476730948686054034624e-02 9.74854715489385070803507460368564e-01
            -1.36155357822940165091196718094579e-01 9.66372208193673642995236150454730e-01
            -3.13099246424719590553564785295748e-01 9.24328073857590082340607295918744e-01
            -4.52497012764203621060232762829401e-01 8.64673336973265316274250835704152e-01
            -6.28598147073706425835837308113696e-01 7.46510479315938835220833880157443e-01
            -7.56196038929077385404298183857463e-01 6.16912535892230939893465802015271e-01
            -8.57531617761826625567778137337882e-01 4.65889526359838046509764808433829e-01
            -9.37925240147350125496927830681670e-01 2.69647492344255668150765359314391e-01
            9.18554492564195745885058386193123e-01 1.44020685669936726602813337194675e-01
            8.54792837875588351614908333431231e-01 3.65805300709686254112540382266161e-01
            7.38374233419929693056360520131420e-01 5.65055577029584865655920111748856e-01
            5.53504183845284658538332678290317e-01 7.47072574905859654315065654373029e-01
            -2.98296256841325657660490833222866e-01 8.80626854499202527115642169519560e-01
            -9.25731351931562973511802283610450e-01 8.66358918789816023764416286212509e-02
            3.47910136167665196005316374794347e-01 8.49253843156532850500184395059478e-01
            1.28040847669653928830868494515016e-01 9.08778902864927240479175907239551e-01
            -7.12947101904133628824666857326520e-02 9.14981211426338147951753398956498e-01
            -4.99445906040914544554709664225811e-01 7.69952816673407758329972239152994e-01
            -6.53977229284602668002435166272335e-01 6.43884567715035882606855466292473e-01
            -7.77626989885058073070922546321526e-01 4.87411343288857157318716417648830e-01
            -8.64556302852384717105849176732590e-01 3.07921990384837473975210286880611e-01
            8.27234149333384460867080179014010e-01 2.37824513512651547486598246905487e-01
            7.43183348431954993174031187663786e-01 4.34229602473535925000192037259694e-01
            6.13350965710830586807844611030305e-01 6.03885278767788080678258211264620e-01
            8.43919509133080780749480709346244e-01 2.62584821936897930272358792080922e-02
            4.37234899512399688426000921026571e-01 7.22298614446212239315059377986472e-01
            2.20641397314511600313835515407845e-01 8.14988968988928252912273819674738e-01
            1.31308290965072935879698334815657e-02 8.44225815235340126818641692807432e-01
            -1.95188624476517941230824249032594e-01 8.21456661460299608634727519529406e-01
            -3.68029386809962277826713261674740e-01 7.59897372167333773340658353845356e-01
            -7.65528996461353528069082585716387e-01 3.56167097525037112948353978936211e-01
            -8.29666380656661539561014251376037e-01 1.56663150072414769642392684545484e-01
            -5.12797852569569001524030227301409e-01 6.48922675472492382908740182756446e-01
            -6.40869261123383715528234461089596e-01 5.22827759868876085214139948220691e-01
            7.61213486761561863858105425606482e-01 1.19351099136822971780169666544680e-01
            6.89616891438978862893804944178555e-01 2.95118890998512295453792830812745e-01
            5.95694443153257036094316845264984e-01 4.55867027957148063244119384762598e-01
            4.83154349693173335023033132529235e-01 5.73784359454594161675800023658667e-01
            3.05807992474722623033045465490432e-01 6.84943857933431043960581519058906e-01
            1.01702305134184642354533423258545e-01 7.21839248652392129201871284749359e-01
            -1.01702305134184559087806576371804e-01 7.21839248652392129201871284749359e-01
            -3.17746066250371439476651858058176e-01 6.56073697954026591183662731054937e-01
            -6.29407335104375853518376970896497e-01 3.67752180525778349373666742394562e-01
            -7.25797138581998901862846196308965e-01 6.79247627219963234779243066441268e-02
            -4.88200678413237776354094421549235e-01 5.11527220777989954925146776076872e-01
            6.26355280389529367468526288575958e-01 1.18272589534903327379566917443299e-01
            -6.06835993296561015952761408698279e-01 1.95088236310941853801637080323417e-01
            5.38557382919158889045263549633091e-01 2.92593577716324937387781801589881e-01
            4.09169480073552860321939306231798e-01 4.56328163708871914483466980527737e-01
            2.32345956671816594729662597274000e-01 5.67159953483874712532042394741438e-01
            2.85862242535767328843743229072061e-02 6.12240054390838395192986354231834e-01
            -1.78487551251604398272476714737422e-01 5.86342263925924478407125661760801e-01
            -3.35056554709036502615049357700627e-01 4.82937478312707557659422263895976e-01
            -4.77668336907012991421339620501385e-01 3.42526586893652285059630457908497e-01
            4.73388141132926376641876231587958e-01 8.93883121286789678050510588036559e-02
            3.82580658051661470775428597335122e-01 2.92777462580812830594823026331142e-01
            2.23371610080909555051675852155313e-01 4.26839227719716107767311541465460e-01
            -4.62999456609476056634377982845763e-01 1.33109374913382788596649675127992e-01
            4.93270439150808107098278298963123e-02 4.51302799229478746045884918203228e-01
            -1.45651851567803608533679948777717e-01 4.29991758047330474568070712848566e-01
            -3.03403495435356607146104579442181e-01 2.98721147700096223420729302233667e-01
            3.30740121671221465327050736959791e-01 7.31734277522316373110200515839097e-02
            2.01653686588664365908485365253000e-01 2.72174887337730320169271180930082e-01
            -5.26081519010394779711781154674100e-02 3.04505969015524291521046507114079e-01
            -2.99498961683887898743705591186881e-01 7.61043675671729119569519639298960e-02
            -1.50610461212588503387976857084141e-01 1.57806725903307659697460962888726e-01
            1.67353065809331846836016666202340e-01 8.42894324342724765575241008264129e-02
            3.19004612710111351558950332218956e-02 1.84645925020215539991141895370674e-01
            -1.54546369837805258606877600868756e-01 -2.42314139574956004252737074011748e-02
            8.16765953413491341272845147614134e-02 -1.68643680075239665860920013074065e-01
            -1.01145065784648699458436738041200e-01 -1.93277389868575233400349588919198e-01
            2.62819181092342868399924782352173e-01 -9.36061712652680316670839033577067e-02
            -2.75987717933174558915254692692542e-01 -1.39004612738443195096849080982793e-01
            5.26798842594551706280947556138017e-03 -3.38696954385553194999403103793156e-01
            2.10020744639087181271364102030930e-01 -2.65771829645936541464124047706719e-01
            -4.46107067402119028542983869556338e-01 -8.42369174866020448133596687512181e-02
            -3.76988255273104588827237648729351e-01 -2.52956971123359131148333744931733e-01
            -2.35098809997527552484442026070610e-01 -3.88376007744440643332239915253012e-01
            4.36317076516773094674306321394397e-01 -1.25438361730452224707121899882623e-01
            -3.74244788498353381234196035620698e-02 -4.80297835611737944372379160995479e-01
            1.68674703298139938567601348040625e-01 -4.51259844189338787945331432638341e-01
            3.43289942834026529094870738845202e-01 -3.37992037864078498365927316626767e-01
            -6.12907053652976596680446164100431e-01 -0.00000000000000000000000000000000e+00
            -5.77379444863542978794157534139231e-01 -2.05640543343572484991099713624862e-01
            -4.74915386516501170088844219208113e-01 -3.87440875576461396523342273212620e-01
            -3.33546582238622713045828049871488e-01 -5.14200091301534545351614724495448e-01
            -1.41689598461817584107436118756596e-01 -5.96304548117236432602794593549334e-01
            6.65936691819491027866106946930813e-02 -6.09278540277316760231940406811191e-01
            2.67156634660980774054195308053750e-01 -5.51617973759187751880972427898087e-01
            5.47396493144276852227392282657092e-01 -2.75702984588343524841036469297251e-01
            6.05509535035596679897196281672223e-01 -9.49381872512224389559776227542898e-02
            4.54218113457841132785119953041431e-01 -4.47208282698271897270814179137233e-01
            -7.16310268152481333103764882253017e-01 -1.35258491430211624795276748045580e-01
            -6.40539921794616939898503460426582e-01 -3.47999810850526669625537579122465e-01
            -5.03294534740448717968774872133508e-01 -5.27342271267012718816147298639407e-01
            -3.37996168496707716233373730574385e-01 -6.45874484591396069710356186988065e-01
            6.93987688285326531634211733035045e-01 -2.23106136789297943012755354175169e-01
            7.28616012387742428835224473004928e-01 -2.26708239118378340748449772945605e-02
            -5.95694443153256925072014382749330e-01 -4.55867027957148007732968153504771e-01
            -1.50627608209164737251839483178628e-01 -7.34831913043614615155263436463429e-01
            3.49854731214880934619948504860076e-02 -7.49294757390319365875086532469140e-01
            2.40654961361221053639525280232192e-01 -7.10458870276373666285962826805189e-01
            4.27587506932606775666982912298408e-01 -6.16307991751940620339667020743946e-01
            6.22883217959981694278326358471531e-01 -4.17951089920782503650542594186845e-01
            -8.43919509133080558704875784314936e-01 -2.62584821936897860883419753008639e-02
            -8.18321780392883058041775257152040e-01 -2.07940158500367677785192199735320e-01
            -7.41905375658127530513752390106674e-01 -4.03070787023952603611576250841608e-01
            -6.37318086336603140118484134291066e-01 -5.53818835550747623486245174717624e-01
            -5.02635898407677972343776673369575e-01 -6.78414916857259520099887595279142e-01
            -3.20074925596417536599602726710145e-01 -7.81307678054544485490851002396084e-01
            1.69547019250400610168583170889178e-01 -8.27129647664645362148405638436088e-01
            3.68029386809962166804410799159086e-01 -7.59897372167333662318355891329702e-01
            7.76235444207844516029126680223271e-01 -3.32186966839438657661531806297717e-01
            8.29666380656661317516409326344728e-01 -1.56663150072414714131241453287657e-01
            -9.35214719029080349121940685108711e-02 -8.55646288687078593326873487967532e-01
            5.33668274643185336358897075115237e-01 -6.75333258243827061484410023695091e-01
            6.83551094276371062896657804230927e-01 -5.23101078726122992179625725839287e-01
            -9.06677730926088831786557875602739e-01 -1.42158521401507087489335390273482e-01
            -8.54561770168299039696080399153288e-01 -3.34660625008630530086151111390791e-01
            -7.62092412294782750592503361986019e-01 -5.11359666073755603221684395975899e-01
            -4.50612088996721560363312164554372e-01 -7.99513788633038346986836586438585e-01
            -2.67263649213790144276714499937952e-01 -8.77977046837371544185657512571197e-01
            9.97160133314542451721251836715965e-02 -9.12321363144382946330779304844327e-01
            2.94439334286677789620512157853227e-01 -8.69240491112305391219194916629931e-01
            -6.20707755030848518096320276526967e-01 -6.92247207690482313502400302240858e-01
            -7.22286145426517811607425301190233e-02 -9.26966742095936657719335016736295e-01
            4.81484404456514136771971834605210e-01 -7.95397436490627329774838472076226e-01
            6.62543814085388649282037931698142e-01 -6.52318946626488593842907448561164e-01
            8.02789198618462540579798769613262e-01 -4.69056303968118870617587390370318e-01
            8.85159401641335441546232232212787e-01 -2.84564838651332085994738463341491e-01
            9.25731351931562862489499821094796e-01 -8.66358918789816023764416286212509e-02
            -9.74028945757079656075916318513919e-01 -6.06723912530364173356645096646389e-02
            -9.37925240147350125496927830681670e-01 -2.69647492344255390595009203025256e-01
            -8.71605914635885548058524818770820e-01 -4.38995052141537223100442588474834e-01
            -7.75016148413103067582596850115806e-01 -5.93096531714634611631709049106576e-01
            -5.31097983868784484506875287479488e-01 -8.18748103975527996567507216241211e-01
            -3.41694270563973445664629480233998e-01 -9.14143616559654037523330316616921e-01
            -1.66143539641541965190540963703825e-01 -9.61670343968445862081750874494901e-01
            1.51772739315890366207373318729879e-02 -9.75798737747193056790706577885430e-01
            1.95970988315750771491252635314595e-01 -9.56038125793923332373935863870429e-01
            3.69958727564431322232252341564163e-01 -9.03074784351726056463860459189164e-01
            6.28598147073706425835837308113696e-01 -7.46510479315938724198531417641789e-01
            7.75016148413102956560294387600152e-01 -5.93096531714634944698616436653538e-01
            8.84836989079309854666632872977061e-01 -4.11675878562335562094176566461101e-01
            9.58970209611948787298274510249030e-01 -1.81079163102287027387049533899699e-01
            -6.55763318872003897475053690868663e-01 -7.31343088137230146195122415520018e-01
            -1.00000000000000000000000000000000e+00 -0.00000000000000000000000000000000e+00
            -9.82635248222263668615994447463891e-01 -1.85547753829493261568117645765597e-01
            -9.31144062097659541699101737322053e-01 -3.64651526282655180644098891207250e-01
            -8.47314704957777498073312472115504e-01 -5.31091132257275000405627451982582e-01
            -7.54822939153257044253564345126506e-01 -6.55928601699939939351224893471226e-01
            -6.20009547460775078953076899779262e-01 -7.84594265246365774046921615081374e-01
            -4.63663531985327337370250688763917e-01 -8.86011359468314774190389471186791e-01
            -2.91214712227252259246768062439514e-01 -9.56657719031419606636745811556466e-01
            -7.76838472890062359255125556956045e-02 -9.96978043825629267615795470192097e-01
            1.08652150085474366592031003619923e-01 -9.94079830940052588950095469044754e-01
            2.91214712227251926179860674892552e-01 -9.56657719031419717659048274072120e-01
            4.63663531985327448392553151279571e-01 -8.86011359468314774190389471186791e-01
            6.20009547460774967930774437263608e-01 -7.84594265246366107113829002628336e-01
            7.34058528759460204859976784064202e-01 -6.79086206865885944061744794453261e-01
            8.47314704957777609095614934631158e-01 -5.31091132257275000405627451982582e-01
            9.19353038982236192566688259830698e-01 -3.93433589966752705890939978417009e-01
            9.76389420563607002279127300425898e-01 -2.16017821976483392898416013849783e-01
            9.98065597133594328838057663233485e-01 -6.21696374314805247363047158160043e-02
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            ];
        
    case   19
        
        % DEGREE.             :          19
        % CARDIN.             :         210
        % LEBESGUE CONST.     : 3.77187e+01
        % ABS.DET.VAND.MATRIX : 2.07061e+205
        % CONDITIONING VAND.  : 6.06969e+02
        % POINTS OUT DISK     :           0
        
        pts=[
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            9.82635248222263668615994447463891e-01 1.85547753829493261568117645765597e-01
            9.52013107532729940807314505946124e-01 3.06057254587887295205206328319036e-01
            9.06672601177072601252859840315068e-01 4.21835032062062242896871566699701e-01
            8.12657913828250366705674423428718e-01 5.82741036046300786566121132636908e-01
            6.90420020571746673176960484852316e-01 7.23408733147249849437798729923088e-01
            5.44204182756027332601433954550885e-01 8.38952804078301195289668612531386e-01
            3.79088403840379339904131938965293e-01 9.25360460617241531444676638784586e-01
            2.00807072855718665715940574045817e-01 9.79630807749081622404219160671346e-01
            1.55518119203507890896398535574008e-02 9.99879063260149303360435624199454e-01
            -1.70243555722398520746452277307981e-01 9.85402015288681321081298847275320e-01
            -3.50126449191390465287554434326012e-01 9.36702444523675126220041420310736e-01
            -4.63663531985327448392553151279571e-01 8.86011359468314774190389471186791e-01
            -5.95308864276666072079535751981894e-01 8.03496954638675653370682994136587e-01
            -7.34058528759460204859976784064202e-01 6.79086206865885944061744794453261e-01
            -8.47314704957777609095614934631158e-01 5.31091132257275000405627451982582e-01
            -9.06672601177072490230557377799414e-01 4.21835032062062353919174029215355e-01
            -9.61070940398724538056285382481292e-01 2.76301732750830475460190882586176e-01
            -9.95649347969018605475355343514821e-01 9.31792674840717094486564064936829e-02
            1.37797673609091619884026158615598e-01 9.78028659751593920645973412320018e-01
            4.55450776097020615384991515384172e-01 8.70317662406518866013982460572151e-01
            2.86056499045453882512646259783651e-01 9.39712680715751536020263756654458e-01
            9.74028945757079767098218781029573e-01 6.06723912530364173356645096646389e-02
            9.19347028040984670660407118703006e-01 3.27436357579940662265727269186755e-01
            8.26908623205479309525856024265522e-01 5.18300738086902912371556340076495e-01
            7.16380022460453180777051329641836e-01 6.62731612081821852555663099337835e-01
            5.80971899178317263690018990018871e-01 7.84146146198620996869976806920022e-01
            -4.55171387624477702393832601046597e-02 9.74854715489385070803507460368564e-01
            -1.95970988315750771491252635314595e-01 9.56038125793923443396238326386083e-01
            -3.69958727564431322232252341564163e-01 9.03074784351726167486162921704818e-01
            -5.31097983868784262462270362448180e-01 8.18748103975528218612112141272519e-01
            -7.16380022460453291799353792157490e-01 6.62731612081821852555663099337835e-01
            -8.42627714430384533450535400334047e-01 4.92333282550383832809615114456392e-01
            -9.19347028040984670660407118703006e-01 3.27436357579940773288029731702409e-01
            -9.64137875772522079387272242456675e-01 1.51167730464734084705469285836443e-01
            1.33975699042927887116860574678867e-01 9.50901927023479309752929111709818e-01
            9.18666028818657220966770182712935e-01 2.03247014449050772411098364500504e-01
            8.43002864973578680540811092214426e-01 3.92212293735222095047987522775657e-01
            7.38374233419929693056360520131420e-01 5.65055577029584865655920111748856e-01
            6.20707755030848629118622739042621e-01 6.92247207690482424524702764756512e-01
            4.81484404456514081260820603347383e-01 7.95397436490627440797140934591880e-01
            2.98296256841325713171642064480693e-01 8.80626854499202527115642169519560e-01
            -4.33651382748694164126490591115726e-02 9.28764651832264420683316075155744e-01
            -2.42970979450412988276397641129734e-01 8.97468337522618564428000809130026e-01
            -4.31103449403852645449575220482075e-01 8.23792528263522116915851256635506e-01
            -5.98878714882386531392910455906531e-01 7.11216281149079865286921631195582e-01
            -7.38374233419929471011755595100112e-01 5.65055577029585087700525036780164e-01
            -8.43002864973578569518508629698772e-01 3.92212293735222206070289985291311e-01
            -9.07823924310096619372245640988695e-01 2.00848291406528628444405626396474e-01
            -9.17310691519927945414281111879973e-01 2.85420424562773979160201776039685e-02
            8.74611549890136341645074935513549e-01 5.44796685771114092000644291147182e-02
            1.22258223485143618924908537337615e-01 8.67736321862669912619026035827119e-01
            8.27234149333384460867080179014010e-01 2.37824513512651547486598246905487e-01
            7.43183348431954993174031187663786e-01 4.34229602473535925000192037259694e-01
            5.33668274643185447381199537630891e-01 6.75333258243826839439805098663783e-01
            3.50910421101204461624689656673581e-01 7.85963557306210591768547146784840e-01
            -1.20086715640734531529432160823490e-01 8.52323892529992965805263338552322e-01
            -3.26297321135257778479399348725565e-01 7.96496638580987403877031738375081e-01
            -4.90650721392566713241478737472789e-01 7.07204854796551596329834410425974e-01
            -6.13350965710830586807844611030305e-01 6.03885278767788080678258211264620e-01
            -7.29319376655605977610719037329545e-01 4.57132457702946415345479636016535e-01
            -8.10848481446325486388104764046147e-01 2.88793312227006937487061577485292e-01
            -8.37801162870634441581785267771920e-01 1.04780042351341293183075720207853e-01
            5.96427128828096786428147879632888e-01 5.18285272393785301225932471425040e-01
            2.06484827470297199747406580172537e-01 7.62698472272620442424795328406617e-01
            1.22883421403891905654415239723676e-02 7.90059453604517236691151538252598e-01
            7.61213486761561863858105425606482e-01 1.19351099136822971780169666544680e-01
            7.08373691321979470636449605080998e-01 3.03145791222202587444911614511511e-01
            3.99009993750634728293391617626185e-01 6.59152245069362319185302112600766e-01
            -1.78124417605785678730967447336297e-01 7.49641480406108562384304150327807e-01
            -3.47799147926150797882627330182004e-01 6.64606928555515175105483649531379e-01
            -5.00765253930148257488497165468289e-01 5.58480776068815409907131197542185e-01
            -6.47662178602191218601547006983310e-01 3.78418180311696272788424266764196e-01
            -7.27006897974432408915390624315478e-01 1.84736534231173199493980519036995e-01
            5.68439690399603958503860212658765e-01 3.81419793159382014557934326148825e-01
            4.56995529337283656889923122434993e-01 5.09666387356552452736480063322233e-01
            6.29577563955036056952963008370716e-01 2.02399293927351564548189344350249e-01
            2.69606011858977701045603225793457e-01 6.03859239879083742330578843393596e-01
            7.18529560444505221994404564611614e-02 6.57396787279586725105673394864425e-01
            6.37115656477336700369562549894908e-01 1.98237982886741888333137495692426e-02
            -1.85627243741411457333967405247677e-01 6.09796580088888595660989722091472e-01
            -3.63352236316084653555691375004244e-01 5.23721772577041111418338914518245e-01
            -4.93912508694651808394127101564663e-01 4.02938923984880403850183938629925e-01
            -5.93533563093091909301790565223200e-01 2.32437630751039125964751974606770e-01
            -6.08169203898790811102514908270678e-01 7.60610008262025960590335671440698e-02
            4.25522375531473595611231530710938e-01 3.25639466495563412173908091062913e-01
            3.05438557953331024652499081639689e-01 4.40247250454620375759162698159344e-01
            1.56040645903459368470933554817748e-01 5.12602839480522987081201335968217e-01
            -2.49911708978592279895014627300043e-02 5.35243678705893288771733296016464e-01
            4.93362430696440923405532430479070e-01 1.25366163403215741256602200337511e-01
            -2.07527844528146404590174256554747e-01 4.64817552050956595000030802111723e-01
            -3.51452384734242340158516526571475e-01 3.68245005687458548671031621779548e-01
            -4.61533904525248672712223196867853e-01 2.14731501933975443829893947622622e-01
            3.22745376284550911094584080274217e-01 2.31434373252114994334860398339515e-01
            6.17636929971615306761423269676925e-03 3.97099860863648679298876231769100e-01
            3.67946484319790545480799437427777e-01 1.14486228866388112940244425885794e-02
            1.80746976429128847208005481661530e-01 3.20696455859777573849100917868782e-01
            -1.70685930308296596180639426165726e-01 3.26162535377816920068028139212402e-01
            -3.05686204710023168296118001308059e-01 2.05113701490846622732533433008939e-01
            -3.65278902906213787105116352904588e-01 4.56837977944827347553946594871377e-02
            2.18522162523189705796156090400473e-01 1.18721204779335190448996684153826e-01
            1.15989934227716615794401278094483e-02 2.48419249112557055658712101831043e-01
            -1.69029840962572425189236469122989e-01 1.37896289403151711683293001442507e-01
            4.10203474245386251473277638979198e-02 8.46977315683590248340806283522397e-02
            -9.18863615112826409569990460113331e-02 -2.03290728729459725665495994917364e-02
            6.90810100186229764096523808802885e-02 -6.39076575260162649261275191747700e-02
            1.07107040330036859532292226049321e-01 -1.90038300557361344855777929296892e-01
            -2.48689887164854850354345217056107e-01 -0.00000000000000000000000000000000e+00
            -2.10718598369071974385491330394871e-01 -1.32076993755316729473037185016437e-01
            -1.93191872168347773819352397595139e-02 -2.47938357224833455516588287537161e-01
            2.25480306882191589012975896366697e-01 -1.04906106525697220832604728002480e-01
            -1.90633161523624167221768743729626e-01 -3.14920123232545867697496078108088e-01
            2.00334921346193101099331101977441e-01 -3.08839125724880758649959489048342e-01
            -3.54698744772832252891703319619410e-01 -1.78648390679101543021545239753323e-01
            -6.17636929971615480233770867357634e-03 -3.97099860863648790321178694284754e-01
            3.74129453397171629625006517017027e-01 -2.03261302825670064908791800917243e-01
            -4.79657731501052064793100271344883e-01 -4.48894544605579828844277301413968e-02
            -3.39830012033788264602662820834666e-01 -3.78996999816961221974054296879331e-01
            -1.92971697785455198470572213409469e-01 -4.71046798952016343431381528716884e-01
            1.48240349399497289839544578171626e-01 -4.86978399684274809722239751863526e-01
            3.27878875744264608460554200064507e-01 -3.89382338826274265741744784463663e-01
            5.02897509708521650573231909220340e-01 -7.88495889533338267796125364839099e-02
            -4.78554956670983466793245497683529e-01 -2.41030097006778604162846590952540e-01
            -4.16250869145060459208806946662662e-02 -5.34207549887516841558010582957650e-01
            5.19325159332512664711600791633828e-01 -3.25509501093029729457128951253253e-01
            -6.29730562493730605666542032849975e-01 -9.87358160368733972500621121071163e-02
            -3.63352236316084764577993837519898e-01 -5.23721772577041000396036452002591e-01
            2.59867316391299729705366416965262e-01 -5.82046665293017406739295438455883e-01
            4.25536695845488366352071807341417e-01 -4.74581777142825067539178007791634e-01
            6.12609673260480214196377346524969e-01 -1.76121352864510583025037249171874e-01
            -4.99173365880521691106963544370956e-01 -4.33773367109321850154657340681297e-01
            -2.12165888280497400009139141729975e-01 -6.26353749144998084297242257889593e-01
            1.12584083394110412434940826642560e-01 -6.51658044824243387616036216059001e-01
            -6.29340462162034763871076847863151e-01 -2.69323825386874715626106535637518e-01
            -3.19275442590909591666914479901607e-02 -6.83802144010186019329466944327578e-01
            -7.50111069630459481594186854636064e-01 -0.00000000000000000000000000000000e+00
            4.27587506932606775666982912298408e-01 -6.16307991751940620339667020743946e-01
            5.50625428179120079974495638452936e-01 -5.09390081003461170361390486505115e-01
            6.47662178602191218601547006983310e-01 -3.78418180311696217277273035506369e-01
            -7.52320978650438565438207660918124e-01 -1.66444592508463440339028238668106e-01
            -6.39824937362150292230467130139004e-01 -4.29318887101863611022167788178194e-01
            -5.14385502756606771690428558940766e-01 -5.73670821854044721277432472561486e-01
            -3.57258891586890392488840006990358e-01 -6.82683485720116611794594518869417e-01
            -1.78124417605785734242118678594125e-01 -7.49641480406108340339699225296499e-01
            2.69777065748030842051718991569942e-01 -7.21741638045945599344577203737572e-01
            7.40517886824198390272044889570680e-01 -2.12894144086411857275109582587902e-01
            7.69022759750360163089055731688859e-01 -4.79025289395251449575674485004129e-02
            6.13822813160358679573924689520936e-02 -7.87767198557331682629012448160211e-01
            -8.54088381272911090213995066733332e-01 -1.06817011872997802424123392484034e-01
            -8.19437691912045407249820527795237e-01 -2.63436341693240150263477517000865e-01
            -7.43183348431955104196333650179440e-01 -4.34229602473535980511343268517521e-01
            -6.31835025983950471228922651789617e-01 -5.84518038208162171720516653294908e-01
            -4.90650721392566879774932431246270e-01 -7.07204854796551485307531947910320e-01
            -3.01368549584691058917940154060489e-01 -8.06259160798857177354648229083978e-01
            -9.35214719029080349121940685108711e-02 -8.55646288687078593326873487967532e-01
            2.24930761886806551697048917048960e-01 -8.30832708436468947432729237334570e-01
            4.22619240503915749584962213702966e-01 -7.49846527368601800667136103584198e-01
            5.54412901166159599597449414432049e-01 -6.58409577748820473708235567755764e-01
            6.83551094276371062896657804230927e-01 -5.23101078726122992179625725839287e-01
            7.80411212566091694320391525252489e-01 -3.63091140558373159930027895825333e-01
            8.45795455360354608842499146703631e-01 -1.59708749737226929354960702767130e-01
            -8.54792837875588240592605870915577e-01 -3.65805300709686198601389151008334e-01
            -5.98878714882386531392910455906531e-01 -7.11216281149079643242316706164274e-01
            -4.31103449403852478916121526708594e-01 -8.23792528263522005893548794119852e-01
            -2.14942308396627035360637592020794e-01 -9.04590580191853899094667212921195e-01
            1.58288454984692406046775658978731e-01 -9.16203622962311059474416197190294e-01
            3.52467483963694316884129875688814e-01 -8.60378397252632431957408698508516e-01
            5.30002589884955188637150058639236e-01 -7.63925106555574440214684273087187e-01
            6.82510359306270797397075966728153e-01 -6.31398387034945551832265664415900e-01
            8.02789198618462540579798769613262e-01 -4.69056303968118870617587390370318e-01
            8.85159401641335441546232232212787e-01 -2.84564838651332085994738463341491e-01
            9.25731351931562862489499821094796e-01 -8.66358918789816023764416286212509e-02
            -9.29524751816569239082355124992318e-01 -1.45740718909438154993196690156765e-01
            -7.47192607105389705601794503309065e-01 -5.71804012993040622347962198546156e-01
            -4.38830463720387420334390071730013e-02 -9.39856850604922744452096594613977e-01
            -9.75916761938747323590348514699144e-01 -0.00000000000000000000000000000000e+00
            -9.29085549226586260296301134076202e-01 -2.98686404865273813502568600597442e-01
            -8.57531617761826514545475674822228e-01 -4.65889526359837990998613577176002e-01
            -6.73792470854062353069480195699725e-01 -7.05986708411275420971264793479349e-01
            -5.31097983868784484506875287479488e-01 -8.18748103975527996567507216241211e-01
            -3.69958727564431211209949879048509e-01 -9.03074784351726056463860459189164e-01
            1.06035454489098929231616352808487e-01 -9.70139169719633542854353436268866e-01
            2.84201318985743800471510667193797e-01 -9.33618303440851060592819976591272e-01
            4.52497012764203565549081531571574e-01 -8.64673336973265205251948373188497e-01
            6.28598147073706425835837308113696e-01 -7.46510479315938724198531417641789e-01
            7.93086479827163404188183903897880e-01 -5.68706744947136910717233604373178e-01
            8.84836989079309854666632872977061e-01 -4.11675878562335562094176566461101e-01
            9.52874801707685126217484139488079e-01 -2.10815413344350444102559549719444e-01
            9.74028945757079656075916318513919e-01 -6.06723912530364173356645096646389e-02
            -7.80075218897435140874563330726232e-01 -5.96968086086889893060458689433290e-01
            -2.27081553883883430211554355082626e-01 -9.55678926643151793740571520174854e-01
            -4.58142608465692061092688902590453e-02 -9.81218271562045551981157132104272e-01
            -9.98065597133594328838057663233485e-01 -6.21696374314805247363047158160043e-02
            -9.69198999199666122450480543193407e-01 -2.46278906832001348581329125408956e-01
            -9.19353038982236303588990722346352e-01 -3.93433589966752372824032590870047e-01
            -8.63421704896663477768470329465345e-01 -5.04482863448639795578287703392562e-01
            -7.54822939153257044253564345126506e-01 -6.55928601699939939351224893471226e-01
            -6.20009547460775078953076899779262e-01 -7.84594265246365774046921615081374e-01
            -4.90994080973322200023289951786865e-01 -8.71162908099950450413473390653962e-01
            -3.20825769815367811688133770076092e-01 -9.47138229311000112708995857246919e-01
            -1.39515338943923122849710694026726e-01 -9.90219910019669469214420587377390e-01
            4.66403903874177058752614755121613e-02 -9.98911744842610804795413059764542e-01
            2.31176322114969662324668320252385e-01 -9.72911870671437473134801621199585e-01
            3.79088403840379450926434401480947e-01 -9.25360460617241531444676638784586e-01
            5.17849624898325422428513320483034e-01 -8.55471662881163830149944260483608e-01
            6.67588140216153647088503930717707e-01 -7.44530775081015172567333593178773e-01
            7.94141650844753299587353012611857e-01 -6.07732703080531955208698491333053e-01
            8.78693400099268773217886518978048e-01 -4.77386540051127017392218476743437e-01
            9.31144062097659652721404199837707e-01 -3.64651526282654958599493966175942e-01
            9.82635248222263668615994447463891e-01 -1.85547753829493428101571339539078e-01
            ];
        
    case   20
        
        % DEGREE.             :          20
        % CARDIN.             :         231
        % LEBESGUE CONST.     : 3.76050e+01
        % ABS.DET.VAND.MATRIX : 2.25868e+230
        % CONDITIONING VAND.  : 7.05914e+02
        % POINTS OUT DISK     :           0
        
        pts=[
            9.87930439740756671795907095656730e-01 1.54898180214084663086637760898157e-01
            9.42034263469989197048448659188580e-01 3.35516685797524916168299569108058e-01
            8.78693400099268773217886518978048e-01 4.77386540051127017392218476743437e-01
            7.94141650844753521631957937643165e-01 6.07732703080531622141791103786090e-01
            6.90420020571746673176960484852316e-01 7.23408733147249849437798729923088e-01
            5.95308864276666072079535751981894e-01 8.03496954638675653370682994136587e-01
            4.35884418475371182566391325963195e-01 9.00002652068530029083603949402459e-01
            2.91214712227252259246768062439514e-01 9.56657719031419606636745811556466e-01
            1.70243555722398631768754739823635e-01 9.85402015288681321081298847275320e-01
            -1.55518119203506659242730592040971e-02 9.99879063260149303360435624199454e-01
            -1.70243555722398520746452277307981e-01 9.85402015288681321081298847275320e-01
            -3.20825769815367700665831307560438e-01 9.47138229311000112708995857246919e-01
            -4.90994080973322089000987489271211e-01 8.71162908099950561435775853169616e-01
            -5.95308864276666072079535751981894e-01 8.03496954638675653370682994136587e-01
            -6.67588140216153647088503930717707e-01 7.44530775081015172567333593178773e-01
            -7.94141650844753299587353012611857e-01 6.07732703080531955208698491333053e-01
            -8.78693400099268773217886518978048e-01 4.77386540051127017392218476743437e-01
            -9.42034263469989086026146196672926e-01 3.35516685797525027190602031623712e-01
            -9.82635248222263668615994447463891e-01 1.85547753829493428101571339539078e-01
            -9.99516282291988078512190440960694e-01 3.10998622698367525873930361512976e-02
            9.80387111455245374713740602601320e-01 6.10684422313683833438702208695759e-02
            9.59094879566002922466338986851042e-01 2.12191552457679288856695620779647e-01
            9.03068769110925928522703998169163e-01 3.86464799432759409381077375655877e-01
            8.32306432034965348876198731886689e-01 5.21684048191385030257549715315690e-01
            7.21056334089076100823945125739556e-01 6.67057723150064729722430456604343e-01
            5.59935419747482732155674511886900e-01 8.07069122601240418468648840644164e-01
            3.72373705991493109124235161289107e-01 9.08969782792743208332808535487857e-01
            1.67228074304831275664895429144963e-01 9.67947836460428034932590435346356e-01
            -7.63078527795451044823948905104771e-02 9.79318821706343545230311065097339e-01
            -2.56693193507894323879270359611837e-01 9.48154442773351369133649768627947e-01
            -4.28163707059645515240475788232288e-01 8.84061130748925005917726593906991e-01
            -5.59935419747482621133372049371246e-01 8.07069122601240640513253765675472e-01
            -6.99962143055805063873719973344123e-01 6.89159808196074696873267839691835e-01
            -8.15679528628174499438330258271890e-01 5.47316315781874784818228363292292e-01
            -9.03068769110925817500401535653509e-01 3.86464799432759742447984763202840e-01
            -9.65230076445349971514531262073433e-01 1.82261192988056625052806225539825e-01
            9.40425648265154401350685020588571e-01 2.92612623268146618271678960354620e-02
            9.04253165421912474819521321478533e-01 2.59966986773986052927654100130894e-01
            7.47192607105389705601794503309065e-01 5.71804012993040622347962198546156e-01
            6.06031099216872815027556953282328e-01 7.19710308506056772870351778692566e-01
            4.36252100510386875420465457864339e-01 8.33631049198726659099634161975700e-01
            2.45872772057113470056677329012018e-01 9.08186765675896290339608185604447e-01
            4.38830463720387420334390071730013e-02 9.39856850604922744452096594613977e-01
            -1.60178887617591858560928130827961e-01 9.27145805873857864476406120957108e-01
            -3.56676988906966196957171177928103e-01 8.70653861745386503656618515378796e-01
            -5.60114661985896211682245393603807e-01 7.55994832532815519421376393438550e-01
            -7.10198387414804788164701676578261e-01 6.17150607146509488920571584458230e-01
            -8.26745721960402857320104885729961e-01 4.49163814891701540510382528736955e-01
            -9.04253165421912474819521321478533e-01 2.59966986773986274972259025162202e-01
            -9.36787324125863407608960642392049e-01 8.76705808410048470502573536577984e-02
            8.30397341244183850861304563295562e-01 4.18239846697198658365124401825597e-01
            8.69532717558211043495930425706320e-01 1.08748565900110033544123666615633e-01
            8.25510917908952479571382809808711e-01 2.94015513030549180406580944691086e-01
            6.24442087874597318020164493646007e-01 6.14805234509524956010295682062861e-01
            4.76889760656918293424411103842431e-01 7.35179946455346011724429899913957e-01
            3.06818146286454151994149697202374e-01 8.20838609349512915436264393065358e-01
            1.22258223485143618924908537337615e-01 8.67736321862669912619026035827119e-01
            -6.80748743108634396525857823689876e-02 8.73658519661462684702257774915779e-01
            -2.55193397651792275304671875346685e-01 8.38325549702758632264476545969956e-01
            -4.30261393018919802155863862935803e-01 7.63405875774425091151442757109180e-01
            -5.85011946789474879260239958966849e-01 6.52437291701728927506565014482476e-01
            -7.12137558478206034173751959315268e-01 5.10659862623055516017700483644148e-01
            -8.05635209378760097642668824846623e-01 3.44768483041503759345403068437008e-01
            -8.61089832063729687661179923452437e-01 1.62596736147919473935985479329247e-01
            7.29319376655605866588416574813891e-01 4.57132457702946415345479636016535e-01
            7.90155012375690413151119173562620e-01 0.00000000000000000000000000000000e+00
            7.71498994688930128660331320133992e-01 1.70687564797198015886792177298048e-01
            5.08947070130900125484174623125000e-01 6.04415273952933818080168748565484e-01
            3.44416258074777548170430918617058e-01 7.11141606683363569629818812245503e-01
            2.06484827470297199747406580172537e-01 7.62698472272620442424795328406617e-01
            3.68531382437769922111314713220054e-02 7.89295122108335633903664074750850e-01
            -1.58668715137436272533122405548056e-01 7.74060193020583264811307344643865e-01
            -3.22133251809616216743137329103774e-01 7.21508913084924929925989545154152e-01
            -4.70386283019887274647885533340741e-01 6.34887146136352331637908719130792e-01
            -5.80020025876412326226727600442246e-01 5.36583370190274799860219445690745e-01
            -6.94303994429875226401804866327439e-01 3.77209367462086275768484711079509e-01
            -7.59395020804670584091411456029164e-01 2.18321199061157156240398080626619e-01
            -7.86717322866308022000225719239097e-01 7.36260652520344532900864464863844e-02
            6.98603246068906114096819237602176e-01 3.25029478470568622405778569373069e-01
            6.11896658615709632123014216631418e-01 4.68266095791476322229840434374637e-01
            -5.49307752067164228870765896317607e-02 7.04969935483201526338348230638076e-01
            6.72660115354069221282884427637327e-01 1.27016177895548393328439829019771e-01
            4.56995529337283656889923122434993e-01 5.09666387356552452736480063322233e-01
            9.55048215067226535790467778497259e-02 6.77852173636931287781237642775523e-01
            -1.99350188459021221420996994311281e-01 6.54877272927299025617742245231057e-01
            -3.72533398339925547748663348102127e-01 5.74302714042559392382258920406457e-01
            -5.30426190175559253425774386414560e-01 4.32727162321924152532659491043887e-01
            -6.29340462162034763871076847863151e-01 2.69323825386874937670711460668826e-01
            5.70991018226140290003911559324479e-01 3.33620503451037164754922059728415e-01
            2.69606011858977701045603225793457e-01 6.03859239879083742330578843393596e-01
            -6.56199840197020378695924591738731e-01 8.20679775750131762501737853199302e-02
            -4.56614197778184224829622905872384e-02 5.86008991020103953140107932995306e-01
            5.61811488168884487492960033705458e-01 1.74807156353659655378951498505558e-02
            5.29501800863655613937908128718846e-01 1.88588352078824822433134045240877e-01
            4.35534300113122552211564197932603e-01 3.55313378699189497123001046929858e-01
            3.20405657181481007267365157531458e-01 4.61820244796354029137575025742990e-01
            1.29940368013814422365470591103076e-01 5.46857590619436839673994654731359e-01
            -2.29152183381391760796930157084716e-01 5.13251400883818487130838548182510e-01
            -3.75240196866717501400501078023808e-01 4.18488372972401989624557927527349e-01
            -4.76261511466449127816247255395865e-01 2.98517497566481826432038815255510e-01
            -5.40202000534838910894563923648093e-01 1.55304611250983493064836693520192e-01
            -3.52677286524265276579370720355655e-02 4.52618560345753262463830424167099e-01
            4.18385739854944760374166889960179e-01 7.90023911770121323172588745364919e-02
            3.38129069498996459586237506300677e-01 2.58759999778555405125501920338138e-01
            1.73583240752560125086390030446637e-01 3.88788970593937910358306453417754e-01
            -2.31710871400610557424570856710488e-01 3.57208730576990318716923411557218e-01
            -3.74129453397171629625006517017027e-01 2.03261302825670037153216185288329e-01
            -3.96955783167380948928126827013330e-01 1.23512446994978649511542911909601e-02
            -4.80577417671124336451260461444690e-03 3.08979622867089354087966057704762e-01
            2.68130244644002591059006590512581e-01 7.70857260207097327864289582066704e-02
            1.72977149400963814418474839840201e-01 2.18894821853120258303704304125858e-01
            -1.98804588303331614129731974571769e-01 1.95736488469920755228770303801866e-01
            -2.45688309585854108840763387888728e-01 3.85216109594820541039972283670068e-02
            1.14832053254705107603328428922396e-01 1.06232487487264465331193719066505e-01
            -3.14131470184789707289496618614066e-02 1.53248021347156898608332653566322e-01
            -1.32549222590790938269833532103803e-01 -8.30809571622772857768524090715800e-02
            2.43283937816634156584005843626528e-03 -1.56415546366028584968077552730392e-01
            1.41834843331769877572412497102050e-01 -6.59895375758573349278535147277580e-02
            -2.45147643563065925631505592718895e-01 -1.33186598817104706604652619716944e-01
            -8.95075363666766044001832369758631e-02 -2.64243142167512989892941277503269e-01
            1.51828126858278816824210366576153e-01 -2.34060370724518018725746060226811e-01
            2.91104596695713091492052626563236e-01 -1.03680357807794693014713516276970e-01
            -3.80269878878140565880983103852486e-01 -1.91527607262398313903162261340185e-01
            -2.63987225881430198626276251161471e-01 -3.34063990422616474695871602307307e-01
            -8.54994932217638392568659355674754e-02 -4.17106511318723893744220276857959e-01
            1.23993193865445841539418836418918e-01 -4.07325045879456282271746658807388e-01
            3.03403495435356662657255810700008e-01 -2.98721147700096223420729302233667e-01
            3.98918455828908502525109724956565e-01 -2.16728953886744291068566781177651e-01
            4.40007137993724928293204357032664e-01 -1.11808283987969783734861550783535e-01
            -5.35567606069838220150813867803663e-01 -1.66641395243349449417635810277716e-02
            -5.29501800863655613937908128718846e-01 -1.88588352078824822433134045240877e-01
            -4.56781505242848717696091398465796e-01 -3.27549049953955007108419295036583e-01
            -3.34613217297944076644711230983376e-01 -4.51632282357207037737367727459059e-01
            -1.80330832399832080570334369440388e-01 -5.32370655224012834239033509220462e-01
            8.74141497591173627368377196944493e-03 -5.62015401320889118608192802639678e-01
            1.63686949128929948305000152686262e-01 -5.37721402161494999027979702077573e-01
            3.20405657181480951756213926273631e-01 -4.61820244796354140159877488258644e-01
            4.46373821599886311251026427271427e-01 -3.41596450578711408052612341634813e-01
            5.59579264596866909897698860731907e-01 -1.79895940603883197628221068953280e-01
            -6.49828348934634703049084691883763e-01 -1.22704931191595922879145064143813e-01
            -6.11379299592122116280279442435130e-01 -3.07928712960629069606000030034920e-01
            -4.87797290388368331992552384690498e-01 -4.80269240869232894031171099413768e-01
            -3.17399528945229825183105276664719e-01 -6.06516511943977931764493405353278e-01
            -1.16539733372777440378875724036334e-01 -6.74554097742164238127315911697224e-01
            1.16539733372777357112148877149593e-01 -6.74554097742164238127315911697224e-01
            2.98383417186724730907343428043532e-01 -6.16094210801656827491967760579428e-01
            4.72625026957614935430029845520039e-01 -4.95207354679488953586030675069196e-01
            5.91052829283025271678297940525226e-01 -3.45342284164384227818089811989921e-01
            6.79255469426510272157315739605110e-01 -8.49514419507671897369149860423931e-02
            6.73174924114871120472969323600410e-01 -2.16415160150432939190068282186985e-01
            -1.13367829894940597146302252440364e-02 -7.28880468332157693822637156699784e-01
            -7.84047013277229476457819146162365e-01 -9.80572514069376760525997838158219e-02
            -7.44353095110453666372052339283982e-01 -2.65110191018593954748183705305564e-01
            -6.82236987918062864899582109501353e-01 -3.98619663211583707962404332647566e-01
            -5.63051791009691826594973917963216e-01 -5.54362358230792717783685930044157e-01
            -4.30005662760491291773945476961671e-01 -6.62902763289110241728963046625722e-01
            -2.76654168793879939869384543271735e-01 -7.40140131644943854993812237808015e-01
            -1.34518798878713430333675660222070e-01 -7.78620341585458231037364384974353e-01
            6.13822813160358679573924689520936e-02 -7.87767198557331682629012448160211e-01
            2.30104764543907380769383053120691e-01 -7.55907891820571165020226089836797e-01
            3.87961434127866044985211146922666e-01 -6.88353738430958927985159334639320e-01
            5.27498115194359074031638101587305e-01 -5.88294723798221896871041280974168e-01
            6.27495005951287265766325162985595e-01 -4.80203041523709517335305463348050e-01
            7.26431411894637446557965176907601e-01 -3.10873523149191810510671984957298e-01
            7.76434166719852325577733154204907e-01 -1.46611487723424943618866223005170e-01
            -8.75882794985077262062134195730323e-01 -2.72530170555022044354132049193140e-02
            -8.55616571564046135556225181062473e-01 -1.89297860406518608789028235150909e-01
            -7.94523257024214446531118483108003e-01 -3.69656856472502515220668328765896e-01
            -6.95911633536319018666915781068383e-01 -5.32560227390583729167872206744505e-01
            -5.64438256239785052237323270674096e-01 -6.70315487201727289168218248960329e-01
            -4.06311450371473925624599132788717e-01 -7.76417672896829169637555878580315e-01
            -2.28998146953339448694819679985812e-01 -8.45856516308431061190731270471588e-01
            -4.08712856563476500948439706917270e-02 -8.75353034779851268964989685628098e-01
            1.49185565113957530058996781008318e-01 -8.63514368526156816940897442691494e-01
            3.32197700612490320892788986384403e-01 -8.10899553087355284830550772312563e-01
            4.99523075713226294425339801819064e-01 -7.19993121022344806370085734670283e-01
            6.43260392295085581260138951620320e-01 -5.95087779402224903435580927180126e-01
            7.56622207695807680316590904112672e-01 -4.42081703207699294733146189173567e-01
            8.34255445620248226035187144589145e-01 -2.68200016671251373345796764624538e-01
            8.72494174606568195784461750008631e-01 -8.16536145378860128696274500725849e-02
            -9.33607640519270698398202057433082e-01 -1.16762129785023635597518421036511e-01
            -8.86341922194870845963521333032986e-01 -3.15681197330148488600087830491248e-01
            -7.97222111146896339484158033883432e-01 -4.99693432902995171573934385378379e-01
            -6.90661552996608252286137030750979e-01 -6.38939152502183027593218866968527e-01
            -5.36332389436897538637083471257938e-01 -7.73048633288231479276930713240290e-01
            -3.29427242746408333040619709208841e-01 -8.81325316284738091709982654720079e-01
            -1.31267299386467778399278927281557e-01 -9.31678870373090495249357445572969e-01
            7.30912379726026961712648244429147e-02 -9.38037468505137583818509483535308e-01
            2.73998322371160285548796764487633e-01 -9.00100850308277578015747621975606e-01
            4.61966888458152535879008837582660e-01 -8.19660426857480750584272755077109e-01
            6.28120842711295979121643995313207e-01 -7.00514688168311083771300218359102e-01
            7.64614202859460689865045424085110e-01 -5.48289834096425732212765069562010e-01
            8.65001594258010375604328601184534e-01 -3.70174098660339734312429982310277e-01
            9.24542607948889649804868895444088e-01 -1.74578313300823101394954051102104e-01
            -9.81812100991156899709721983526833e-01 -3.05489982070789797830023815095046e-02
            -9.52031820332836620224270518519916e-01 -2.41916630304473467827364174809190e-01
            -8.77295493834631789908939936140087e-01 -4.41860678768291437190640635890304e-01
            -7.80075218897435140874563330726232e-01 -5.96968086086889893060458689433290e-01
            -6.55763318872003897475053690868663e-01 -7.31343088137230146195122415520018e-01
            -5.08677084332258666599102525651688e-01 -8.40318907807838022527846533193951e-01
            -3.72373705991493109124235161289107e-01 -9.08969782792743208332808535487857e-01
            -1.97250227522319382389426323243242e-01 -9.62278852872969991949503310024738e-01
            -1.52763465750910282930918882016158e-02 -9.82168456110988907781234047433827e-01
            1.67228074304831164642592966629309e-01 -9.67947836460428034932590435346356e-01
            2.86056499045453549445738872236689e-01 -9.39712680715751647042566219170112e-01
            4.00462417288204752807700970151927e-01 -8.96949327043514488266851003572810e-01
            5.84764307624744406055583567649592e-01 -7.89264814540898584915851188270608e-01
            7.41452949687800955125283053348539e-01 -6.44310302838147164194992910779547e-01
            8.63129324216954429971337958704680e-01 -4.68930711961702606505042467688327e-01
            9.35150338056000673958578772726469e-01 -3.00636139174706551457916248182300e-01
            9.74694044904629741665758047020063e-01 -1.21900622523338497682487968631904e-01
            -9.92269872363276461690873020415893e-01 -1.24098752613259338817286447920196e-01
            -9.69198999199666122450480543193407e-01 -2.46278906832001348581329125408956e-01
            -9.06672601177072601252859840315068e-01 -4.21835032062062242896871566699701e-01
            -8.30387982764797527579503366723657e-01 -5.57185604695428549071323232055875e-01
            -7.54822939153257044253564345126506e-01 -6.55928601699939939351224893471226e-01
            -6.20009547460775078953076899779262e-01 -7.84594265246365774046921615081374e-01
            -4.63663531985327337370250688763917e-01 -8.86011359468314774190389471186791e-01
            -2.61321923212860740548535432026256e-01 -9.65251704193435644540954854164738e-01
            -7.76838472890062359255125556956045e-02 -9.96978043825629267615795470192097e-01
            7.76838472890061110254222853654937e-02 -9.96978043825629267615795470192097e-01
            2.31176322114969662324668320252385e-01 -9.72911870671437473134801621199585e-01
            4.07683614941688698962707348982804e-01 -9.13123250228618843493677559308708e-01
            5.17849624898325422428513320483034e-01 -8.55471662881163830149944260483608e-01
            6.44110411450397712940230121603236e-01 -7.64932531574647689431856178998714e-01
            7.54822939153256822208959420095198e-01 -6.55928601699940050373527355986880e-01
            8.47314704957777609095614934631158e-01 -5.31091132257275000405627451982582e-01
            9.19353038982236192566688259830698e-01 -3.93433589966752705890939978417009e-01
            9.69198999199666011428178080677753e-01 -2.46278906832001820426114591100486e-01
            9.98065597133594328838057663233485e-01 -6.21696374314805247363047158160043e-02
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            ];
        
end



stats_matrix=[
    1    3 1.795357e+00 1.83419e+00 3.023e+00   0
    2    6 2.157895e+00 3.67886e+01 7.805e+00   0
    3   10 2.763623e+00 4.71597e+03 1.488e+01   0
    4   15 3.382679e+00 6.66841e+06 2.587e+01   0
    5   21 3.881098e+00 8.40775e+10 3.416e+01   0
    6   28 4.842268e+00 1.68891e+16 4.987e+01   0
    7   36 5.488209e+00 4.08613e+22 6.153e+01   0
    8   45 6.646185e+00 1.81221e+30 8.287e+01   0
    9   55 8.069821e+00 1.02172e+39 9.051e+01   0
    10   66 7.805513e+00 2.42794e+49 1.154e+02   0
    11   78 1.179652e+01 6.02208e+60 1.432e+02   0
    12   91 2.053252e+01 5.07760e+73 1.475e+02   0
    13  105 2.105305e+01 1.43982e+88 2.703e+02   0
    14  120 1.728409e+01 8.45765e+103 2.113e+02   0
    15  136 1.664044e+01 6.54684e+121 2.285e+02   0
    16  153 2.409251e+01 2.59044e+140 2.769e+02   0
    17  171 2.886381e+01 1.07727e+160 4.042e+02   0
    18  190 3.235490e+01 9.95608e+181 3.822e+02   0
    19  210 3.771867e+01 2.07061e+205 6.070e+02   0
    20  231 3.760500e+01 2.25868e+230 7.059e+02   0
    ];


