
function [pts,stats_matrix]=pointset_disk_DLP(deg)

%--------------------------------------------------------------------------
% OBJECT:
%--------------------------------------------------------------------------
% INPUT:
% deg: DEGREE OF THE INTERPOLATORY SET.
%--------------------------------------------------------------------------
% OUTPUT:
%  pts: MATRIX N x 2 WHOSE COMPONENTS ARE THE CARTESIAN COORDINATES OF
%       EACH POINT IN THE POINTSET.
%
%  stats_matrix: MATRIX WITH STATISTICS WITH COLUMNS THAT CONSIST OF
%       DEGREE, CARDINALITY, LEBESGUE CONSTANT, ABS. VALUE OF VANDERMONDE
%       MATRIX "V" DETERMINANT, CONDITIONING OF "V", POINTS OUT OF THE
%       DOMAIN.
%--------------------------------------------------------------------------
% Copyright (C) 2018
% Gerard Meurant, Alvise Sommariva.
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation; either version 2 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%
% Authors:
% * Gerard Meurant    <gerard.meurant@gmail.com>
% * Alvise Sommariva  <alvise@math.unipd.it>
%
% Date: February 10, 2018.
%--------------------------------------------------------------------------
% STATISTICS.
%--------------------------------------------------------------------------
% |DEGR|  Card |  Leb. Const. |    Abs.Det.   |  Cond.Vand. | Out |
%--------------------------------------------------------------------------
%   1  |     3 | 3.000232e+00 | 1.436523e+00  | 4.04690e+00 |   0 |
%   2  |     6 | 3.041392e+00 | 2.847323e+01  | 8.27490e+00 |   0 |
%   3  |    10 | 4.038372e+00 | 1.992523e+03  | 2.54068e+01 |   0 |
%   4  |    15 | 5.773425e+00 | 3.510344e+06  | 3.05649e+01 |   0 |
%   5  |    21 | 9.257063e+00 | 1.129516e+10  | 3.94016e+01 |   0 |
%   6  |    28 | 9.874270e+00 | 2.143637e+15  | 7.20701e+01 |   0 |
%   7  |    36 | 1.107126e+01 | 5.805822e+21  | 7.79548e+01 |   0 |
%   8  |    45 | 1.692800e+01 | 4.264179e+28  | 1.55877e+02 |   0 |
%   9  |    55 | 1.945967e+01 | 1.130448e+37  | 1.57270e+02 |   0 |
%  10  |    66 | 1.900962e+01 | 1.125781e+47  | 1.24567e+02 |   0 |
%  11  |    78 | 2.512112e+01 | 2.944595e+57  | 2.27507e+02 |   0 |
%  12  |    91 | 3.573396e+01 | 1.092766e+71  | 2.94785e+02 |   0 |
%  13  |   105 | 3.731040e+01 | 6.566734e+84  | 2.19799e+02 |   0 |
%  14  |   120 | 2.974765e+01 | 9.557869e+100 | 3.21073e+02 |   0 |
%  15  |   136 | 5.673142e+01 | 2.441863e+116 | 7.47865e+02 |   0 |
%  16  |   153 | 5.647200e+01 | 1.142277e+135 | 5.60368e+02 |   0 |
%  17  |   171 | 5.800516e+01 | 1.031227e+154 | 6.87427e+02 |   0 |
%  18  |   190 | 7.475907e+01 | 1.117985e+176 | 1.36738e+03 |   0 |
%  19  |   210 | 1.483895e+02 | 2.541071e+198 | 2.18408e+03 |   0 |
%  20  |   231 | 1.061188e+02 | 4.683133e+222 | 1.49841e+03 |   0 |
%--------------------------------------------------------------------------
% CAPTION.
%--------------------------------------------------------------------------
% Card.      : cardinality of the point set
% Leb. Const.: Lebesgue Constant (evaluated on a WAM of degree  500.
% Abs.Det.   : Absolute value of the determinant of the Vandermonde matrix
%           (on the point set).
% Cond.Vand. : Conditioning of the Vandermonde matrix
%           (on the point set). We used Koornwinder IIb polynomial basis.
% Out        : Points outside the domain.
%
% The polynomial basis is the orthonormal Koornwinder type II.
% The cardinality of the WAM is 250501.
%--------------------------------------------------------------------------

switch deg
    case    1
        
        % DEGREE.             :           1
        % CARDIN.             :           3
        % LEBESGUE CONST.     : 3.00023e+00
        % ABS.DET.VAND.MATRIX : 1.43652e+00
        % CONDITIONING VAND.  : 4.04690e+00
        % POINTS OUT DISK     :           0
        
        pts=[
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            -1.00000000000000000000000000000000e+00 -0.00000000000000000000000000000000e+00
            1.55518119203506659242730592040971e-02 -9.99879063260149303360435624199454e-01
            ];
        
    case    2
        
        % DEGREE.             :           2
        % CARDIN.             :           6
        % LEBESGUE CONST.     : 3.04139e+00
        % ABS.DET.VAND.MATRIX : 2.84732e+01
        % CONDITIONING VAND.  : 8.27490e+00
        % POINTS OUT DISK     :           0
        
        pts=[
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            -1.00000000000000000000000000000000e+00 -0.00000000000000000000000000000000e+00
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            1.55518119203506659242730592040971e-02 -9.99879063260149303360435624199454e-01
            -7.12583964147506909725393597909715e-01 7.01586839984776666767629649257287e-01
            4.63663531985327337370250688763917e-01 8.86011359468314774190389471186791e-01
            ];
        
    case    3
        
        % DEGREE.             :           3
        % CARDIN.             :          10
        % LEBESGUE CONST.     : 4.03837e+00
        % ABS.DET.VAND.MATRIX : 1.99252e+03
        % CONDITIONING VAND.  : 2.54068e+01
        % POINTS OUT DISK     :           0
        
        pts=[
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            -1.00000000000000000000000000000000e+00 -0.00000000000000000000000000000000e+00
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            -5.77379444863542978794157534139231e-01 -2.05640543343572484991099713624862e-01
            3.20825769815367700665831307560438e-01 -9.47138229311000112708995857246919e-01
            -6.44110411450397712940230121603236e-01 7.64932531574647689431856178998714e-01
            8.30387982764797527579503366723657e-01 5.57185604695428549071323232055875e-01
            7.76838472890062359255125556956045e-02 9.96978043825629267615795470192097e-01
            -5.44204182756027332601433954550885e-01 -8.38952804078301195289668612531386e-01
            1.88576656052852786871198986773379e-01 -5.56713883071412340974859489506343e-01
            ];
        
    case    4
        
        % DEGREE.             :           4
        % CARDIN.             :          15
        % LEBESGUE CONST.     : 5.77343e+00
        % ABS.DET.VAND.MATRIX : 3.51034e+06
        % CONDITIONING VAND.  : 3.05649e+01
        % POINTS OUT DISK     :           0
        
        pts=[
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            -1.00000000000000000000000000000000e+00 -0.00000000000000000000000000000000e+00
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            -5.77379444863542978794157534139231e-01 -2.05640543343572484991099713624862e-01
            6.58434727513669493248471553670242e-01 6.16205551893828779030926057203033e-02
            -3.50126449191390465287554434326012e-01 9.36702444523675126220041420310736e-01
            6.44110411450397712940230121603236e-01 -7.64932531574647689431856178998714e-01
            -8.30387982764797305534898441692349e-01 5.57185604695428882138230619602837e-01
            2.31176322114969773346970782768039e-01 9.72911870671437362112499158683931e-01
            1.55518119203506659242730592040971e-02 -9.99879063260149303360435624199454e-01
            8.30387982764797527579503366723657e-01 5.57185604695428549071323232055875e-01
            -6.67588140216153647088503930717707e-01 -7.44530775081015172567333593178773e-01
            -1.71171713097775263223709885096468e-01 5.62309298738425056285450409632176e-01
            4.72625026957614935430029845520039e-01 -4.95207354679488953586030675069196e-01
            -2.19620352234453697359839452474262e-01 -6.48360733789267840165848610922694e-01
            ];
        
    case    5
        
        % DEGREE.             :           5
        % CARDIN.             :          21
        % LEBESGUE CONST.     : 9.25706e+00
        % ABS.DET.VAND.MATRIX : 1.12952e+10
        % CONDITIONING VAND.  : 3.94016e+01
        % POINTS OUT DISK     :           0
        
        pts=[
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            -1.00000000000000000000000000000000e+00 -0.00000000000000000000000000000000e+00
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            -5.77379444863542978794157534139231e-01 -2.05640543343572484991099713624862e-01
            6.58434727513669493248471553670242e-01 6.16205551893828779030926057203033e-02
            -8.39358674760744216669650086259935e-01 2.98947556019653504222333140205592e-01
            3.20825769815367700665831307560438e-01 -9.47138229311000112708995857246919e-01
            8.12657913828250144661069498397410e-01 -5.82741036046301008610726057668217e-01
            -8.47314704957777498073312472115504e-01 -5.31091132257275000405627451982582e-01
            -3.79088403840379450926434401480947e-01 9.25360460617241531444676638784586e-01
            -4.66403903874176087307468208109640e-02 -9.98911744842610804795413059764542e-01
            1.70243555722398631768754739823635e-01 9.85402015288681321081298847275320e-01
            5.95308864276666072079535751981894e-01 8.03496954638675653370682994136587e-01
            -5.17849624898325311406210857967380e-01 -8.55471662881163830149944260483608e-01
            -8.12657913828250144661069498397410e-01 5.82741036046301008610726057668217e-01
            -2.75979611538887448229928622822626e-01 4.89666190044305504880384205534938e-01
            3.33546582238622657534676818613661e-01 5.14200091301534434329312261979794e-01
            5.61542946535004583452632687112782e-01 -4.29731915497074734577864774109912e-01
            -6.92574870022565902694111628079554e-02 -6.33650331966511326520219427038683e-01
            -6.07125549517514184749700234533520e-01 -5.61659009956570809407594424556009e-01
            -1.15390426653849978433363787644339e-01 8.18991651837173151307069929316640e-01
            ];
        
    case    6
        
        % DEGREE.             :           6
        % CARDIN.             :          28
        % LEBESGUE CONST.     : 9.87427e+00
        % ABS.DET.VAND.MATRIX : 2.14364e+15
        % CONDITIONING VAND.  : 7.20701e+01
        % POINTS OUT DISK     :           0
        
        pts=[
            1.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            -1.00000000000000000000000000000000e+00 -0.00000000000000000000000000000000e+00
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            -5.77379444863542978794157534139231e-01 -2.05640543343572484991099713624862e-01
            6.58434727513669493248471553670242e-01 6.16205551893828779030926057203033e-02
            -8.39358674760744216669650086259935e-01 2.98947556019653504222333140205592e-01
            8.69969343870673927909820122295059e-01 -1.92473288722008084583592335548019e-01
            -3.20825769815367700665831307560438e-01 9.47138229311000112708995857246919e-01
            4.90994080973322089000987489271211e-01 -8.71162908099950561435775853169616e-01
            -8.47314704957777498073312472115504e-01 -5.31091132257275000405627451982582e-01
            8.93115016186791565999669728626031e-01 4.49828376007635810651663632597774e-01
            7.76838472890062359255125556956045e-02 9.96978043825629267615795470192097e-01
            -6.44110411450397712940230121603236e-01 7.64932531574647689431856178998714e-01
            -4.66403903874176087307468208109640e-02 -9.98911744842610804795413059764542e-01
            5.95308864276666072079535751981894e-01 8.03496954638675653370682994136587e-01
            -5.44204182756027332601433954550885e-01 -8.38952804078301195289668612531386e-01
            7.94141650844753299587353012611857e-01 -6.07732703080531955208698491333053e-01
            -9.19353038982236192566688259830698e-01 3.93433589966752705890939978417009e-01
            -2.75979611538887448229928622822626e-01 4.89666190044305504880384205534938e-01
            4.25536695845488366352071807341417e-01 -4.74581777142825067539178007791634e-01
            1.56040645903459368470933554817748e-01 5.12602839480522987081201335968217e-01
            -8.27234149333384571889382641529664e-01 -2.37824513512651575242173862534401e-01
            -7.18529560444504944438648408322479e-02 -6.57396787279586503061068469833117e-01
            7.01119762849604533094804992288118e-01 4.70447365732077016353684939531377e-01
            -5.21096269117211519095178573479643e-01 -6.18843417594140854376405513903592e-01
            -8.98640826888600224853220765908191e-02 8.22184117448658047244691715604858e-01
            4.45734935818216548852888081455603e-01 -7.36340413152767436777423881721916e-01
            -4.84614100071371178835022419661982e-01 1.55795818176090145046330803779711e-01
            ];
        
    case    7
        
        % DEGREE.             :           7
        % CARDIN.             :          36
        % LEBESGUE CONST.     : 1.10713e+01
        % ABS.DET.VAND.MATRIX : 5.80582e+21
        % CONDITIONING VAND.  : 7.79548e+01
        % POINTS OUT DISK     :           0
        
        pts=[
            9.99516282291988078512190440960694e-01 3.10998622698369156513997779711644e-02
            -1.00000000000000000000000000000000e+00 -0.00000000000000000000000000000000e+00
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            -5.77379444863542756749552609107923e-01 2.05640543343572512746675329253776e-01
            6.58418280570842839871659180062125e-01 2.57847567004490074715761238621781e-01
            -8.39358674760744216669650086259935e-01 2.98947556019653504222333140205592e-01
            8.69602986211723272269580320426030e-01 -2.50005282436187747929778879552032e-01
            3.05438557953331024652499081639689e-01 4.40247250454620375759162698159344e-01
            4.07683614941688698962707348982804e-01 -9.13123250228618843493677559308708e-01
            -7.34058528759460093837674321548548e-01 -6.79086206865885944061744794453261e-01
            8.93115016186791565999669728626031e-01 4.49828376007635810651663632597774e-01
            -3.20825769815367811688133770076092e-01 -9.47138229311000112708995857246919e-01
            -1.08652150085474366592031003619923e-01 9.94079830940052588950095469044754e-01
            -9.31144062097659541699101737322053e-01 -3.64651526282655180644098891207250e-01
            6.90420020571746673176960484852316e-01 -7.23408733147249849437798729923088e-01
            1.55518119203506659242730592040971e-02 -9.99879063260149303360435624199454e-01
            5.95308864276666072079535751981894e-01 8.03496954638675653370682994136587e-01
            -5.95308864276666072079535751981894e-01 8.03496954638675653370682994136587e-01
            2.61321923212860740548535432026256e-01 9.65251704193435644540954854164738e-01
            -8.63421704896663477768470329465345e-01 5.04482863448639795578287703392562e-01
            9.31144062097659652721404199837707e-01 -3.64651526282654958599493966175942e-01
            3.32217728672285328439528484523180e-01 -4.20406630505861300672876268436085e-01
            -5.87172973628806271761959578725509e-01 -3.93989719459664544665145058388589e-01
            6.29407335104375631473772045865189e-01 -3.67752180525778238351364279878908e-01
            -2.39630416464007983678996538401407e-01 -5.36720380009751751870794578280766e-01
            -8.49315557290494571596184414374875e-01 -2.15815851210783088420797071194102e-01
            -1.32796099920397564497775988456851e-01 6.47841476801061078027998973993817e-01
            4.25573334582037887674488274569740e-01 1.32416773250224953373255232236261e-02
            -5.08947070130900014461872160609346e-01 6.04415273952933929102471211081138e-01
            7.12137558478206256218356884346576e-01 5.10659862623055293973095558612840e-01
            -1.15390426653850061700090634531080e-01 -8.18991651837173040284767466800986e-01
            3.13536654749127952079845726984786e-01 7.65347661178281235550002747913823e-01
            -5.43318508130850141846224232722307e-01 -6.87545195759497462795195588114439e-01
            5.43318508130850030823921770206653e-01 -6.87545195759497573817498050630093e-01
            -2.02581355337319857223832286763354e-01 8.52569171363352196202356481080642e-01
            -8.07529152755189910672939390678948e-02 2.98279180445151337330855767504545e-01
            ];
        
    case    8
        
        % DEGREE.             :           8
        % CARDIN.             :          45
        % LEBESGUE CONST.     : 1.69280e+01
        % ABS.DET.VAND.MATRIX : 4.26418e+28
        % CONDITIONING VAND.  : 1.55877e+02
        % POINTS OUT DISK     :           0
        
        pts=[
            9.99516282291988078512190440960694e-01 3.10998622698369156513997779711644e-02
            -1.00000000000000000000000000000000e+00 -0.00000000000000000000000000000000e+00
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            -5.77379444863542756749552609107923e-01 2.05640543343572512746675329253776e-01
            6.58418280570842839871659180062125e-01 2.57847567004490074715761238621781e-01
            -8.39358674760744216669650086259935e-01 2.98947556019653504222333140205592e-01
            8.69602986211723272269580320426030e-01 -2.50005282436187747929778879552032e-01
            3.05438557953331024652499081639689e-01 4.40247250454620375759162698159344e-01
            -3.21613039335863215573141360437148e-01 -3.58680436377076972132016408068012e-01
            3.50126449191390465287554434326012e-01 -9.36702444523675126220041420310736e-01
            -7.34058528759460093837674321548548e-01 -6.79086206865885944061744794453261e-01
            8.78693400099268773217886518978048e-01 4.77386540051127017392218476743437e-01
            -2.61321923212860574015081738252775e-01 9.65251704193435644540954854164738e-01
            -9.31144062097659541699101737322053e-01 -3.64651526282655180644098891207250e-01
            6.67588140216153647088503930717707e-01 -7.44530775081015172567333593178773e-01
            1.55518119203507890896398535574008e-02 9.99879063260149303360435624199454e-01
            9.76389420563607002279127300425898e-01 -2.16017821976483392898416013849783e-01
            -7.76838472890062359255125556956045e-02 -9.96978043825629267615795470192097e-01
            5.44204182756027332601433954550885e-01 8.38952804078301195289668612531386e-01
            -7.12583964147506909725393597909715e-01 7.01586839984776666767629649257287e-01
            -4.90994080973322200023289951786865e-01 -8.71162908099950450413473390653962e-01
            8.47314704957777609095614934631158e-01 -5.31091132257275000405627451982582e-01
            -9.06672601177072490230557377799414e-01 4.21835032062062353919174029215355e-01
            3.20825769815367811688133770076092e-01 9.47138229311000112708995857246919e-01
            2.90170027744525160784405670710839e-01 -4.18239785228382954240800017942092e-01
            -3.05438557953330969141347850381862e-01 4.40247250454620486781465160674998e-01
            6.29407335104375631473772045865189e-01 -3.67752180525778238351364279878908e-01
            -7.53311370448327854987269347475376e-01 -2.95009281787430799326443775498774e-01
            8.63562631523739243633031037461478e-01 2.19436112757292411945542198736803e-01
            -9.01818044416109243499590775172692e-01 -1.70287309362290029124409329597256e-01
            -3.19275442590909591666914479901607e-02 -6.83802144010186019329466944327578e-01
            3.97147890634780564855788043132634e-01 0.00000000000000000000000000000000e+00
            -5.71032987105135059202609681960894e-01 5.98317310446660699518872661428759e-01
            6.66952077006153332128235433629015e-01 5.44106390357654068345993891853141e-01
            -3.66366063854022649959318869150593e-01 -7.00086316705688549433261869126000e-01
            -1.10238744369834135161667632019089e-01 7.82427225256246972051599186670501e-01
            4.90650721392566768752629968730616e-01 -7.07204854796551707352136872941628e-01
            -7.35314864832400161276382277719676e-01 -5.27279853996768688162433136312757e-01
            7.18560849174356808255481610103743e-01 -5.49892990415564275608062416722532e-01
            2.70882156675796514910814494214719e-01 7.99695256317808578216954629169777e-01
            -3.67946484319790545480799437427777e-01 1.14486228866387523134262593771382e-02
            -7.49214253156778569459062211421951e-03 4.81695412382958321195758344401838e-01
            -1.40717001403969123318749012696571e-02 -9.04717625632165622917568725824822e-01
            -4.05275082863474578331874909054022e-01 -8.36801303130384366291139031091006e-01
            -1.58288454984692406046775658978731e-01 9.16203622962311170496718659705948e-01
            ];
        
    case    9
        
        % DEGREE.             :           9
        % CARDIN.             :          55
        % LEBESGUE CONST.     : 1.94597e+01
        % ABS.DET.VAND.MATRIX : 1.13045e+37
        % CONDITIONING VAND.  : 1.57270e+02
        % POINTS OUT DISK     :           0
        
        pts=[
            9.99516282291988078512190440960694e-01 3.10998622698369156513997779711644e-02
            -1.00000000000000000000000000000000e+00 -0.00000000000000000000000000000000e+00
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            -5.77379444863542756749552609107923e-01 2.05640543343572512746675329253776e-01
            6.58418280570842839871659180062125e-01 2.57847567004490074715761238621781e-01
            -8.39358674760744216669650086259935e-01 2.98947556019653504222333140205592e-01
            8.69602986211723272269580320426030e-01 -2.50005282436187747929778879552032e-01
            3.05438557953331024652499081639689e-01 4.40247250454620375759162698159344e-01
            -3.21613039335863215573141360437148e-01 -3.58680436377076972132016408068012e-01
            -9.43618424191435267722738444717834e-01 -1.78180336394002186706231327661953e-01
            4.07683614941688698962707348982804e-01 -9.13123250228618843493677559308708e-01
            -7.34058528759460093837674321548548e-01 -6.79086206865885944061744794453261e-01
            -2.00807072855718332649033186498855e-01 9.79630807749081733426521623187000e-01
            9.06672601177072601252859840315068e-01 4.21835032062062242896871566699701e-01
            -9.52013107532729829785012043430470e-01 3.06057254587887683783264947123826e-01
            6.90420020571746673176960484852316e-01 -7.23408733147249849437798729923088e-01
            -5.17849624898325311406210857967380e-01 -8.55471662881163830149944260483608e-01
            1.08652150085474366592031003619923e-01 -9.94079830940052588950095469044754e-01
            9.76389420563607002279127300425898e-01 -2.16017821976483392898416013849783e-01
            1.08652150085474491492121273950033e-01 9.94079830940052588950095469044754e-01
            -6.20009547460774967930774437263608e-01 7.84594265246366107113829002628336e-01
            5.95308864276666072079535751981894e-01 8.03496954638675653370682994136587e-01
            -2.61321923212860740548535432026256e-01 -9.65251704193435644540954854164738e-01
            8.63421704896663477768470329465345e-01 -5.04482863448639795578287703392562e-01
            -8.12657913828250144661069498397410e-01 5.82741036046301008610726057668217e-01
            3.79088403840379339904131938965293e-01 9.25360460617241531444676638784586e-01
            -9.31144062097659541699101737322053e-01 -3.64651526282655180644098891207250e-01
            2.63606906204042312769075806500041e-01 -4.35470506407352020428191963219433e-01
            -2.91599183060326117011840096893138e-01 4.49533392147918209325041516422061e-01
            -7.35748147910326832921157347300323e-01 -2.88131231262685805205592259881087e-01
            6.59117646225687803251958030159585e-01 -3.58092928184935099800156876881374e-01
            8.27234149333384460867080179014010e-01 2.37824513512651547486598246905487e-01
            -5.43627368783207276514701788983075e-01 -4.16021663071997005456381657495513e-01
            9.14112569320840728792632035037968e-03 -5.87714167460328562597737800388131e-01
            1.02845977502097878780729089953638e-02 6.61231888422635250002201701136073e-01
            -5.89364354285963787383195722213713e-01 5.80268846518084191465902677009581e-01
            5.82940903694401435508609665703261e-01 6.10794194948258262201079560327344e-01
            3.38737920245291479481863916589646e-01 0.00000000000000000000000000000000e+00
            -7.89772800404129893259153050166788e-01 2.45737120567051277320036462015196e-02
            8.27080574274561830527829897619085e-01 0.00000000000000000000000000000000e+00
            -2.83258247575991717326360230799764e-01 -7.57808196292209412803231316502206e-01
            5.12407358530913237260051573684905e-01 -6.91603597427189575874706406466430e-01
            1.40805137833424715010366412570875e-01 -8.15006864696273036763329855602933e-01
            -6.34916961087438669331106666504638e-01 -6.25118451711136446924399479030399e-01
            7.72075220538887330157251653872663e-01 5.18058073521236006619972158659948e-01
            1.95188624476518052253126711548248e-01 8.21456661460299497612425057013752e-01
            -2.50660741695854949462329841480823e-01 8.23435504228073433807821857044473e-01
            -4.52015345049808614508179971380741e-01 -4.23025022104585521343089737911214e-02
            7.38374233419929359989453132584458e-01 -5.65055577029585087700525036780164e-01
            -4.31126107096531444029530177886045e-02 3.05994780364509166847852839055122e-01
            -2.87017620950650365063694380296511e-01 1.79900705601546362366605080751469e-01
            4.56514751199307122764281530180597e-01 8.09986787329361690446205557236681e-01
            -6.28120842711295979121643995313207e-01 7.00514688168311083771300218359102e-01
            3.56676988906966196957171177928103e-01 -8.70653861745386503656618515378796e-01
            -1.86705694541296179922440501286474e-01 -9.10837689896810243084246394573711e-01
            ];
        
    case   10
        
        % DEGREE.             :          10
        % CARDIN.             :          66
        % LEBESGUE CONST.     : 1.90096e+01
        % ABS.DET.VAND.MATRIX : 1.12578e+47
        % CONDITIONING VAND.  : 1.24567e+02
        % POINTS OUT DISK     :           0
        
        pts=[
            9.99516282291988078512190440960694e-01 3.10998622698369156513997779711644e-02
            -1.00000000000000000000000000000000e+00 -0.00000000000000000000000000000000e+00
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            -5.77379444863542756749552609107923e-01 2.05640543343572512746675329253776e-01
            6.58418280570842839871659180062125e-01 2.57847567004490074715761238621781e-01
            -8.39358674760744216669650086259935e-01 2.98947556019653504222333140205592e-01
            8.69602986211723494314185245457338e-01 2.50005282436187581396325185778551e-01
            3.05438557953331024652499081639689e-01 4.40247250454620375759162698159344e-01
            -3.21613039335863215573141360437148e-01 -3.58680436377076972132016408068012e-01
            -9.43618424191435267722738444717834e-01 -1.78180336394002186706231327661953e-01
            9.52031820332836509201968056004262e-01 -2.41916630304473939672149640500720e-01
            2.31176322114969662324668320252385e-01 -9.72911870671437473134801621199585e-01
            -7.34058528759460093837674321548548e-01 -6.79086206865885944061744794453261e-01
            7.94141650844753299587353012611857e-01 -6.07732703080531955208698491333053e-01
            -2.61321923212860574015081738252775e-01 9.65251704193435644540954854164738e-01
            5.44204182756027110556829029519577e-01 -8.38952804078301306311971075047040e-01
            -9.52013107532729829785012043430470e-01 3.06057254587887683783264947123826e-01
            -7.76838472890062359255125556956045e-02 -9.96978043825629267615795470192097e-01
            9.61070940398724538056285382481292e-01 2.76301732750830253415585957554867e-01
            -5.70032258213782783151657440612325e-01 -8.21622312620400530036590680538211e-01
            4.63663531985327337370250688763917e-01 8.86011359468314774190389471186791e-01
            7.76838472890062359255125556956045e-02 9.96978043825629267615795470192097e-01
            -6.90420020571746673176960484852316e-01 7.23408733147249849437798729923088e-01
            7.34058528759460093837674321548548e-01 6.79086206865885944061744794453261e-01
            -8.63421704896663477768470329465345e-01 5.04482863448639795578287703392562e-01
            -4.90994080973322089000987489271211e-01 8.71162908099950561435775853169616e-01
            8.63421704896663477768470329465345e-01 5.04482863448639795578287703392562e-01
            -3.20825769815367811688133770076092e-01 -9.47138229311000112708995857246919e-01
            -9.31144062097659541699101737322053e-01 -3.64651526282655180644098891207250e-01
            9.50596473416901455166794221440796e-01 2.95777266676101845133128875886541e-02
            2.63606906204042312769075806500041e-01 -4.35470506407352020428191963219433e-01
            -3.05438557953330969141347850381862e-01 4.40247250454620486781465160674998e-01
            -7.35748147910326832921157347300323e-01 -2.88131231262685805205592259881087e-01
            7.53311370448327966009571809991030e-01 -2.95009281787430632792990081725293e-01
            5.25175296451347972670475883205654e-01 -4.01900847492371537938993242278229e-01
            -5.43627368783207276514701788983075e-01 -4.16021663071997005456381657495513e-01
            -9.14112569320848014631231137627765e-03 -5.87714167460328562597737800388131e-01
            -9.39577682361783184106229782628361e-01 1.47316923654866260839924052561400e-01
            5.13732499562104330537160024050536e-02 6.59313409849052511901845718966797e-01
            -5.21096269117211519095178573479643e-01 6.18843417594141076421010438934900e-01
            5.82940903694401435508609665703261e-01 6.10794194948258262201079560327344e-01
            3.09016994374947451262869435595348e-01 0.00000000000000000000000000000000e+00
            -7.89772800404129893259153050166788e-01 2.45737120567051277320036462015196e-02
            8.08625658528685731241125722590368e-01 2.51603170990182915811583796994455e-02
            -2.92092635341748618760249200931867e-01 -7.13002489246688742063895460887579e-01
            -2.59553500014057525469013398833340e-01 7.66250923534795069791414334758883e-01
            4.81294154046871980323629713893752e-01 -6.93718662760950843804153009841684e-01
            -6.15168742759693354393846220773412e-01 -6.44561900889041528017742166412063e-01
            1.91201445253530166734634576641838e-01 -8.04676508713470628464392575551756e-01
            7.45820559514570491366214355366537e-01 -5.34813281407368168451910150906770e-01
            -7.87813288817082901793753535457654e-01 -4.93796046636581664746046271829982e-01
            2.70882156675796514910814494214719e-01 7.99695256317808578216954629169777e-01
            -3.67946484319790545480799437427777e-01 1.14486228866387523134262593771382e-02
            5.86648238815607458995771139598219e-01 -3.65423960225943547652782683599071e-02
            -5.83241597269268496894767395133385e-01 -7.29434166139658335392681465236819e-02
            1.36281566725891863794961622602386e-02 8.76200702370869888291338156705024e-01
            4.31126107096531790974225373247464e-02 3.05994780364509166847852839055122e-01
            -1.81695671846369116719799308157235e-01 -8.86396456280507227809550840902375e-01
            6.28120842711295979121643995313207e-01 7.00514688168311083771300218359102e-01
            -6.70456548131556528957730733964127e-01 6.60109565493041783312833103991579e-01
            2.79078638742878915035561249169405e-01 -6.25075878300198839454537846904714e-01
            -2.98296256841325657660490833222866e-01 8.80626854499202527115642169519560e-01
            -1.57988688392623199963082925023627e-02 -3.38369286956581138969823996376363e-01
            2.73998322371160563104552920776769e-01 9.00100850308277466993445159459952e-01
            -2.48653459370298035580049145210069e-01 2.30032249381013992328703920975386e-01
            1.60178887617591858560928130827961e-01 -9.27145805873857864476406120957108e-01
            ];
        
    case   11
        
        % DEGREE.             :          11
        % CARDIN.             :          78
        % LEBESGUE CONST.     : 2.51211e+01
        % ABS.DET.VAND.MATRIX : 2.94460e+57
        % CONDITIONING VAND.  : 2.27507e+02
        % POINTS OUT DISK     :           0
        
        pts=[
            9.99516282291988078512190440960694e-01 3.10998622698369156513997779711644e-02
            -1.00000000000000000000000000000000e+00 -0.00000000000000000000000000000000e+00
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            -5.77379444863542756749552609107923e-01 2.05640543343572512746675329253776e-01
            6.58418280570842839871659180062125e-01 2.57847567004490074715761238621781e-01
            -8.39358674760744216669650086259935e-01 2.98947556019653504222333140205592e-01
            8.69602986211723494314185245457338e-01 2.50005282436187581396325185778551e-01
            3.05438557953331024652499081639689e-01 4.40247250454620375759162698159344e-01
            -3.21613039335863215573141360437148e-01 -3.58680436377076972132016408068012e-01
            -9.43618424191435267722738444717834e-01 -1.78180336394002186706231327661953e-01
            9.52031820332836509201968056004262e-01 -2.41916630304473939672149640500720e-01
            4.79169353634575578837484499672428e-01 -8.50182484394046311670933846471598e-01
            1.70243555722398520746452277307981e-01 -9.85402015288681321081298847275320e-01
            -7.34058528759460093837674321548548e-01 -6.79086206865885944061744794453261e-01
            7.94141650844753299587353012611857e-01 -6.07732703080531955208698491333053e-01
            -3.50126449191390465287554434326012e-01 9.36702444523675126220041420310736e-01
            -4.90994080973322200023289951786865e-01 -8.71162908099950450413473390653962e-01
            -9.52013107532729829785012043430470e-01 3.06057254587887683783264947123826e-01
            9.61070940398724538056285382481292e-01 2.76301732750830253415585957554867e-01
            4.90994080973322200023289951786865e-01 8.71162908099950450413473390653962e-01
            -7.76838472890061110254222853654937e-02 9.96978043825629267615795470192097e-01
            6.67588140216153647088503930717707e-01 -7.44530775081015172567333593178773e-01
            -8.63421704896663477768470329465345e-01 -5.04482863448639795578287703392562e-01
            -1.70243555722398631768754739823635e-01 -9.85402015288681321081298847275320e-01
            7.34058528759460093837674321548548e-01 6.79086206865885944061744794453261e-01
            -5.70032258213782561107052515581017e-01 8.21622312620400530036590680538211e-01
            2.00807072855718665715940574045817e-01 9.79630807749081622404219160671346e-01
            -7.94141650844753299587353012611857e-01 6.07732703080531955208698491333053e-01
            -9.82635248222263668615994447463891e-01 -1.85547753829493261568117645765597e-01
            8.63421704896663477768470329465345e-01 5.04482863448639795578287703392562e-01
            5.08056724576624652023326689231908e-01 -3.16469202548081279813274591106165e-02
            -3.20825769815367811688133770076092e-01 -9.47138229311000112708995857246919e-01
            9.59829174616289138022295901464531e-01 2.98650013631469937736007125295146e-02
            1.97887384972315449882529492242611e-01 -4.08592653779509373812572903261753e-01
            -6.89616891438978751871502481662901e-01 -2.95118890998512239942641599554918e-01
            7.35748147910326943943459809815977e-01 -2.88131231262685638672138566107606e-01
            -3.05438557953330969141347850381862e-01 4.40247250454620486781465160674998e-01
            -1.02219116659162986837650066718197e-01 -4.98672653289272349574900999868987e-01
            -8.27234149333384571889382641529664e-01 -2.37824513512651575242173862534401e-01
            8.69969343870673927909820122295059e-01 -1.92473288722008084583592335548019e-01
            4.87797290388368331992552384690498e-01 4.80269240869232894031171099413768e-01
            -9.39577682361783184106229782628361e-01 1.47316923654866260839924052561400e-01
            -1.02845977502097063460695380854304e-02 6.61231888422635250002201701136073e-01
            5.00765253930148146466194702952635e-01 -5.58480776068815298884828735026531e-01
            -5.58561529899243902619332402537111e-01 5.85249958995376462489446112158475e-01
            -3.44416258074777548170430918617058e-01 -7.11141606683363569629818812245503e-01
            7.24083503154279606306431560369674e-01 5.19226065029542827033992580254562e-01
            -8.08625658528685731241125722590368e-01 2.51603170990181597421742054621063e-02
            2.78451424844005535064184186921921e-01 1.73447759090666342451925174827920e-02
            8.43919509133080780749480709346244e-01 2.62584821936897930272358792080922e-02
            -3.13536654749128063102148189500440e-01 7.65347661178281235550002747913823e-01
            1.62456334530964308093814452149672e-01 -7.92537971682264030803821697190870e-01
            7.01111672302254462785242594691226e-01 -5.71974140880407988518641104747076e-01
            -6.24442087874597318020164493646007e-01 -6.14805234509524956010295682062861e-01
            3.99094688368858530047589283640264e-01 7.62627213497277023357412417681189e-01
            -8.02789198618462540579798769613262e-01 -4.69056303968118870617587390370318e-01
            -6.42508010276486296596942793257767e-02 -8.24581173026430636063821566494880e-01
            1.20086715640734628673946815524687e-01 8.52323892529992965805263338552322e-01
            -3.66522970851886398246932685651700e-01 3.43015761620598658154790427943226e-02
            5.30424081977676986632275202282472e-01 -7.15921028748545018594029443192994e-01
            -9.35214719029079100121037981807604e-02 8.55646288687078482304571025451878e-01
            -7.29048150760613578036384296865435e-01 5.94765008436461628527069933625171e-01
            6.84215978371582655981342213635799e-01 -2.12893207115975695298271119781930e-02
            -5.26081519010394779711781154674100e-02 3.04505969015524291521046507114079e-01
            4.54632532267640054879365152373794e-01 2.28981273367617255143358079294558e-01
            -4.31103449403852478916121526708594e-01 -8.23792528263522005893548794119852e-01
            -6.29730562493730605666542032849975e-01 -9.87358160368733972500621121071163e-02
            7.44090387252807961004918979597278e-01 6.07036620270123283837904182291823e-01
            1.29718081570789040002367187298660e-01 -9.20683188194668766790584868431324e-01
            -3.52467483963694316884129875688814e-01 8.60378397252632542979711161024170e-01
            4.66747282280254138786546036499203e-01 -3.13184766777788670122362191250431e-01
            -1.86705694541296179922440501286474e-01 -9.10837689896810243084246394573711e-01
            2.12286389057221441811762474571879e-01 6.97373464354432370804204310843488e-01
            -2.87017620950650198530240686523030e-01 -1.79900705601546306855453849493642e-01
            1.15989934227716893350157434383618e-02 -2.48419249112557166681014564346697e-01
            -1.47906520670461495009506691644674e-02 9.50941498620659242391184307052754e-01
            2.41640042843397184757492368589737e-01 -5.89846956762327345735741346288705e-01
            -1.92583744552694641205548009565973e-01 -6.32649100648937978697006201400654e-01
            ];
        
    case   12
        
        % DEGREE.             :          12
        % CARDIN.             :          91
        % LEBESGUE CONST.     : 3.57340e+01
        % ABS.DET.VAND.MATRIX : 1.09277e+71
        % CONDITIONING VAND.  : 2.94785e+02
        % POINTS OUT DISK     :           0
        
        pts=[
            9.99516282291988078512190440960694e-01 3.10998622698369156513997779711644e-02
            -1.00000000000000000000000000000000e+00 -0.00000000000000000000000000000000e+00
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            -5.77379444863542756749552609107923e-01 2.05640543343572512746675329253776e-01
            6.58418280570842839871659180062125e-01 2.57847567004490074715761238621781e-01
            -8.39358674760744216669650086259935e-01 2.98947556019653504222333140205592e-01
            8.69602986211723494314185245457338e-01 2.50005282436187581396325185778551e-01
            3.05438557953331024652499081639689e-01 4.40247250454620375759162698159344e-01
            -3.21613039335863215573141360437148e-01 -3.58680436377076972132016408068012e-01
            -9.43618424191435267722738444717834e-01 -1.78180336394002186706231327661953e-01
            9.52031820332836509201968056004262e-01 -2.41916630304473939672149640500720e-01
            4.79169353634575578837484499672428e-01 -8.50182484394046311670933846471598e-01
            -7.12583964147506909725393597909715e-01 7.01586839984776666767629649257287e-01
            2.00807072855718332649033186498855e-01 -9.79630807749081733426521623187000e-01
            -7.34058528759460093837674321548548e-01 -6.79086206865885944061744794453261e-01
            7.94141650844753299587353012611857e-01 -6.07732703080531955208698491333053e-01
            -3.50126449191390465287554434326012e-01 9.36702444523675126220041420310736e-01
            -9.61070940398724538056285382481292e-01 2.76301732750830475460190882586176e-01
            9.61070940398724538056285382481292e-01 2.76301732750830253415585957554867e-01
            4.90994080973322200023289951786865e-01 8.71162908099950450413473390653962e-01
            -1.08652150085474366592031003619923e-01 9.94079830940052588950095469044754e-01
            -8.63421704896663477768470329465345e-01 -5.04482863448639795578287703392562e-01
            6.67588140216153647088503930717707e-01 -7.44530775081015172567333593178773e-01
            -5.44204182756027332601433954550885e-01 -8.38952804078301195289668612531386e-01
            9.87930439740756671795907095656730e-01 -1.54898180214084690842213376527070e-01
            1.39515338943923122849710694026726e-01 9.90219910019669469214420587377390e-01
            -5.44204182756027110556829029519577e-01 8.38952804078301306311971075047040e-01
            -1.08652150085474491492121273950033e-01 -9.94079830940052588950095469044754e-01
            7.54822939153257044253564345126506e-01 6.55928601699939939351224893471226e-01
            -8.78693400099268773217886518978048e-01 4.77386540051127017392218476743437e-01
            -2.91214712227252259246768062439514e-01 -9.56657719031419606636745811556466e-01
            6.20009547460775078953076899779262e-01 7.84594265246365774046921615081374e-01
            -9.69198999199666122450480543193407e-01 -2.46278906832001348581329125408956e-01
            5.08056724576624541001024226716254e-01 3.16469202548081210424335552033881e-02
            9.59829174616289138022295901464531e-01 2.98650013631469937736007125295146e-02
            -7.07106781186547461715008466853760e-01 -0.00000000000000000000000000000000e+00
            -1.85084488083002235425666981427639e-01 4.14549280695089950032894421383389e-01
            2.62172364504237476179326904457412e-01 -4.04168595762658289238800080056535e-01
            7.16411900403769119627384043269558e-01 -3.33315064979498643094046883561532e-01
            -8.27234149333384571889382641529664e-01 -2.37824513512651575242173862534401e-01
            -5.25175296451348083692778345721308e-01 -4.01900847492371371405539548504748e-01
            8.49315557290494571596184414374875e-01 -2.15815851210783504754431305627804e-01
            4.36647992887537125805685889190499e-02 5.60384786517919342152538320078747e-01
            -9.39577682361783184106229782628361e-01 1.47316923654866260839924052561400e-01
            5.02497641444559550727433361316798e-01 -4.64866497586133076147518750076415e-01
            9.19353038982236192566688259830698e-01 -3.93433589966752705890939978417009e-01
            -1.02845977502097844086259570417496e-02 -6.61231888422635027957596776104765e-01
            4.96295601832293875599333432546700e-01 5.89390645408275459793401296337834e-01
            -6.24299389990488284851721800805535e-01 5.42505804577096739649277878925204e-01
            -3.88444736040215177208523300578236e-01 6.41698764082337724801163858501241e-01
            -7.01119762849604311050200067256810e-01 -4.70447365732076905331382477015723e-01
            -2.75623806073096055957449834750150e-01 4.32152146213913845862464313540841e-02
            2.78451424844005535064184186921921e-01 1.73447759090666342451925174827920e-02
            7.12137558478206256218356884346576e-01 5.10659862623055293973095558612840e-01
            -8.76306680043863583939867112349020e-01 -0.00000000000000000000000000000000e+00
            -3.60511535149943385114568172866711e-01 -7.44374710321468402440814315923490e-01
            3.06688961077340005445535098260734e-01 -7.48632338561977528357260780467186e-01
            6.54050938260793635059542339149630e-01 -6.05070240803835979193081584526226e-01
            -1.82665129661720521125900518200069e-01 -7.68751191210845674639529079286149e-01
            -8.98640826888600224853220765908191e-02 8.22184117448658047244691715604858e-01
            7.87813288817083012816055997973308e-01 -4.93796046636581664746046271829982e-01
            -8.26745721960402857320104885729961e-01 -4.49163814891701540510382528736955e-01
            8.70671973886632089012493906921009e-01 4.05085517686529172909359886034508e-01
            3.50910421101204461624689656673581e-01 7.85963557306210591768547146784840e-01
            -3.99975241484437707417676222121372e-02 -3.65945193097666932047218324441928e-01
            -5.05106460634242471208210645272629e-01 -6.31714047231086289757584495418996e-02
            1.43741388233179556710439328526263e-01 8.32002439354197265153345597354928e-01
            7.28616012387742428835224473004928e-01 -2.26708239118378340748449772945605e-02
            -7.89745902061648075154209891479695e-01 5.29915000131443103548178896744503e-01
            -3.93328228792179102057957607030403e-01 3.63872585739391518622198873345042e-01
            6.92168147585572879343374097516062e-02 -8.88313941521192140804430437128758e-01
            -4.50612088996721560363312164554372e-01 7.99513788633038569031441511469893e-01
            5.76470298255244650853512666799361e-01 7.29497298788840842220793092565145e-01
            -6.49602919856937188747281197720440e-01 -6.80641365111786522668069210340036e-01
            2.68055981024716671878849183485727e-01 6.54328656744358694297147849283647e-01
            4.61533904525248783734525659383507e-01 -2.14731501933975499341045178880449e-01
            -1.72815288489860968690337017505954e-01 6.38332405006994840768186350032920e-01
            -2.42970979450413127054275719274301e-01 -8.97468337522618453405698346614372e-01
            1.17144759157150307737893513149174e-01 2.85952108253356929168376154848374e-01
            -4.65076024826899192543550043410505e-01 -5.88532843529876004140533041208982e-01
            6.63005986219993403985029090108583e-01 -6.94684838604860632749193882773397e-01
            2.98296256841325602149339601965039e-01 -8.80626854499202416093339707003906e-01
            -2.32345956671816650240813828531827e-01 -5.67159953483874823554344857257092e-01
            -5.74635921652868697506733042246196e-01 4.12060138264013775533101124892710e-01
            7.30912379726028071935672869585687e-02 9.38037468505137583818509483535308e-01
            -2.17509355715542362919023844369804e-01 9.15394069002036014914835959643824e-01
            3.60534496724321473770658030844061e-01 8.80070095991912326738315641705412e-01
            -4.40970223463071908653887476248201e-01 -8.42646876933868416159612024785019e-01
            -1.87716831567467601349363803819870e-01 -1.63122809944959090699967418913729e-01
            4.15956578633134510525337645958643e-01 2.43036472987736090756172302462801e-01
            1.54190104399152844161235975661839e-01 -1.95120659294310994091858901811065e-01
            ];
        
    case   13
        
        % DEGREE.             :          13
        % CARDIN.             :         105
        % LEBESGUE CONST.     : 3.73104e+01
        % ABS.DET.VAND.MATRIX : 6.56673e+84
        % CONDITIONING VAND.  : 2.19799e+02
        % POINTS OUT DISK     :           0
        
        pts=[
            9.99516282291988078512190440960694e-01 3.10998622698369156513997779711644e-02
            -1.00000000000000000000000000000000e+00 -0.00000000000000000000000000000000e+00
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            -5.77379444863542756749552609107923e-01 2.05640543343572512746675329253776e-01
            6.58418280570842839871659180062125e-01 2.57847567004490074715761238621781e-01
            -8.39358674760744216669650086259935e-01 2.98947556019653504222333140205592e-01
            8.69602986211723494314185245457338e-01 2.50005282436187581396325185778551e-01
            3.05438557953331024652499081639689e-01 4.40247250454620375759162698159344e-01
            -3.21613039335863215573141360437148e-01 -3.58680436377076972132016408068012e-01
            -9.43618424191435267722738444717834e-01 -1.78180336394002186706231327661953e-01
            9.52031820332836509201968056004262e-01 -2.41916630304473939672149640500720e-01
            4.79169353634575578837484499672428e-01 -8.50182484394046311670933846471598e-01
            -7.12583964147506909725393597909715e-01 7.01586839984776666767629649257287e-01
            -1.56040645903459479493236017333402e-01 -5.12602839480523320148108723515179e-01
            2.00807072855718332649033186498855e-01 -9.79630807749081733426521623187000e-01
            -7.34058528759460093837674321548548e-01 -6.79086206865885944061744794453261e-01
            7.94141650844753299587353012611857e-01 -6.07732703080531955208698491333053e-01
            -2.61321923212860574015081738252775e-01 9.65251704193435644540954854164738e-01
            -9.52013107532729829785012043430470e-01 3.06057254587887683783264947123826e-01
            9.61070940398724538056285382481292e-01 2.76301732750830253415585957554867e-01
            5.17849624898325311406210857967380e-01 8.55471662881163830149944260483608e-01
            -5.70032258213782783151657440612325e-01 -8.21622312620400530036590680538211e-01
            6.44110411450397712940230121603236e-01 -7.64932531574647689431856178998714e-01
            -8.63421704896663477768470329465345e-01 -5.04482863448639795578287703392562e-01
            -7.76838472890061110254222853654937e-02 9.96978043825629267615795470192097e-01
            9.87930439740756671795907095656730e-01 -1.54898180214084690842213376527070e-01
            -9.87930439740756671795907095656730e-01 1.54898180214084690842213376527070e-01
            1.39515338943923122849710694026726e-01 9.90219910019669469214420587377390e-01
            -4.90994080973322089000987489271211e-01 8.71162908099950561435775853169616e-01
            7.54822939153257044253564345126506e-01 6.55928601699939939351224893471226e-01
            -3.50126449191390798354461821872974e-01 -9.36702444523675015197738957795082e-01
            8.78693400099268773217886518978048e-01 4.77386540051127017392218476743437e-01
            -8.30387982764797305534898441692349e-01 5.57185604695428882138230619602837e-01
            -1.55518119203507890896398535574008e-02 -9.99879063260149303360435624199454e-01
            3.50126449191390798354461821872974e-01 9.36702444523675015197738957795082e-01
            9.50596473416901455166794221440796e-01 2.95777266676101845133128875886541e-02
            -9.61070940398724538056285382481292e-01 -2.76301732750830253415585957554867e-01
            5.06826753681083208569191356218653e-01 4.74321062386743250161025287070515e-02
            -7.28968627421411330402634121128358e-01 -0.00000000000000000000000000000000e+00
            -1.97887384972315477638105107871525e-01 4.08592653779509484834875365777407e-01
            2.62172364504237476179326904457412e-01 -4.04168595762658289238800080056535e-01
            7.52237928764343588561303022288484e-01 -2.41832673786562230011654150985123e-01
            -8.49315557290494571596184414374875e-01 -2.15815851210783088420797071194102e-01
            -5.74635921652868808529035504761850e-01 -4.12060138264013553488496199861402e-01
            8.69969343870673927909820122295059e-01 -1.92473288722008084583592335548019e-01
            -3.88073617299704909378021966404049e-01 4.06616024295136824218133142494480e-01
            4.71240230530109938111849032793543e-01 -4.63967701836859036657045862739324e-01
            -9.39577682361783184106229782628361e-01 1.47316923654866260839924052561400e-01
            9.19353038982236192566688259830698e-01 -3.93433589966752705890939978417009e-01
            6.10715675309401045955937092912791e-02 -5.58755749229459808269382392609259e-01
            3.08438435665264638896321258698663e-02 6.60592189275570840223394952772651e-01
            4.96295601832293875599333432546700e-01 5.89390645408275459793401296337834e-01
            -6.40869261123383715528234461089596e-01 5.22827759868876085214139948220691e-01
            -7.29319376655605977610719037329545e-01 -4.57132457702946470856630867274362e-01
            7.66673266998060709553897140722256e-01 -4.80545623791190978923992815907695e-01
            -1.85932833107239903203122821651050e-01 -2.32537874031211594383350416137546e-02
            2.78451424844005535064184186921921e-01 1.73447759090666342451925174827920e-02
            -3.97222555644926644813352822893648e-01 -7.04785597521960349709502224868629e-01
            9.38749830407518248698295337817399e-01 2.38541602098643473750527732590854e-01
            -9.04827052466019354781678885046858e-01 -0.00000000000000000000000000000000e+00
            -3.06688961077340060956686329518561e-01 7.48632338561977639379563242982840e-01
            2.59553500014057469957862167575513e-01 -7.66250923534794958769111872243229e-01
            6.90402737742399907894252919504652e-01 5.63237681502255060905781647306867e-01
            -5.85011946789474879260239958966849e-01 -6.52437291701728927506565014482476e-01
            -1.28626015340937080444483342489548e-02 -8.26980549846315304840516091644531e-01
            5.64438256239784941215020808158442e-01 -6.70315487201727400190520711475983e-01
            -7.72075220538887108112646728841355e-01 5.18058073521236339686879546206910e-01
            8.26745721960402857320104885729961e-01 4.49163814891701540510382528736955e-01
            -4.59486788675915858171094896533759e-01 7.08351280661530635640588116075378e-01
            1.20086715640734628673946815524687e-01 8.52323892529992965805263338552322e-01
            -4.25573334582037943185639505827567e-01 -1.32416773250224970720489992004332e-02
            3.50910421101204461624689656673581e-01 7.85963557306210591768547146784840e-01
            -8.62296885495731801896113211114425e-01 -4.01188956044199296346874916707748e-01
            7.28616012387742650879829398036236e-01 2.26708239118379624443821995782855e-02
            -1.95188624476517996741975480290421e-01 -8.21456661460299275567820131982444e-01
            -6.68657521810039345844245417538332e-02 8.58140902320898679711547174520092e-01
            -5.83241597269268496894767395133385e-01 -7.29434166139658335392681465236819e-02
            2.40056290007311171064152688359172e-02 3.08083158560810588255662878509611e-01
            4.13127232027148683091155589863774e-01 -7.89441901791273603627985266939504e-01
            4.98931938457512991202946750490810e-01 -1.95390058612234462920298483368242e-01
            -6.28120842711295979121643995313207e-01 7.00514688168311083771300218359102e-01
            -4.05275082863474578331874909054022e-01 -8.36801303130384366291139031091006e-01
            7.62609212839280070639347286487464e-01 -5.83601877347615261015789656084962e-01
            3.17399528945229880694256507922546e-01 -6.06516511943977931764493405353278e-01
            -2.40056290007310893508396532070037e-02 -3.08083158560810255188755490962649e-01
            -2.98296256841325657660490833222866e-01 8.80626854499202527115642169519560e-01
            2.47576786495999651149801934479910e-01 6.62348650476706435341611722833477e-01
            6.34914050953935604759692523657577e-01 7.08090845223080744830213006935082e-01
            -7.06630329000015655616095955338096e-01 2.51674780062448311124256861148751e-01
            4.88089809950044417607983859852538e-01 3.27505481229636741247190911963116e-01
            1.46324007582510282055032746484358e-02 -9.40766981901440146884851856157184e-01
            4.35884418475371071544088863447541e-01 -9.00002652068530029083603949402459e-01
            -3.05686204710023168296118001308059e-01 2.05113701490846622732533433008939e-01
            -6.84289881285495948937125376687618e-01 -6.73729412391420812156184183550067e-01
            2.45872772057113303523223635238537e-01 -9.08186765675896290339608185604447e-01
            1.47906520670462657274235596105427e-02 9.50941498620659242391184307052754e-01
            3.32990041030756189766037778099417e-01 8.90856963693840708806703787558945e-01
            -3.93684815478506644481626608467195e-01 -5.31362069853976159983233173988992e-01
            6.39824937362150292230467130139004e-01 4.29318887101863666533319019436021e-01
            1.82552932689589525461215657742287e-01 -1.68881872160686452577493810167653e-01
            -3.36108577158403865059455029040691e-01 5.96352047532241358318572110874811e-01
            5.23435725027283221821861047828861e-02 4.78901610905782204596903284254950e-01
            -7.68286721159375241008149259869242e-02 -7.02920589498487902169188146217493e-01
            2.62319653052075663079989453763119e-01 2.58271341638852736100062656987575e-01
            4.70386283019887274647885533340741e-01 -6.34887146136352331637908719130792e-01
            ];
        
    case   14
        
        % DEGREE.             :          14
        % CARDIN.             :         120
        % LEBESGUE CONST.     : 2.97477e+01
        % ABS.DET.VAND.MATRIX : 9.55787e+100
        % CONDITIONING VAND.  : 3.21073e+02
        % POINTS OUT DISK     :           0
        
        pts=[
            9.99516282291988078512190440960694e-01 3.10998622698369156513997779711644e-02
            -1.00000000000000000000000000000000e+00 -0.00000000000000000000000000000000e+00
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            -5.77379444863542756749552609107923e-01 2.05640543343572512746675329253776e-01
            6.58418280570842839871659180062125e-01 2.57847567004490074715761238621781e-01
            -8.39358674760744216669650086259935e-01 2.98947556019653504222333140205592e-01
            8.69602986211723494314185245457338e-01 2.50005282436187581396325185778551e-01
            3.05438557953331024652499081639689e-01 4.40247250454620375759162698159344e-01
            -3.21613039335863215573141360437148e-01 -3.58680436377076972132016408068012e-01
            -9.43618424191435267722738444717834e-01 -1.78180336394002186706231327661953e-01
            9.52031820332836509201968056004262e-01 -2.41916630304473939672149640500720e-01
            4.79169353634575578837484499672428e-01 -8.50182484394046311670933846471598e-01
            -7.12583964147506909725393597909715e-01 7.01586839984776666767629649257287e-01
            -1.56040645903459479493236017333402e-01 -5.12602839480523320148108723515179e-01
            7.74474761895350805041005060047610e-01 6.31824506806498931155147147364914e-01
            2.00807072855718332649033186498855e-01 -9.79630807749081733426521623187000e-01
            -7.34058528759460093837674321548548e-01 -6.79086206865885944061744794453261e-01
            7.94141650844753299587353012611857e-01 -6.07732703080531955208698491333053e-01
            -2.31176322114969662324668320252385e-01 9.72911870671437473134801621199585e-01
            -9.52013107532729829785012043430470e-01 3.06057254587887683783264947123826e-01
            9.61070940398724538056285382481292e-01 2.76301732750830253415585957554867e-01
            4.90994080973322200023289951786865e-01 8.71162908099950450413473390653962e-01
            -5.70032258213782783151657440612325e-01 -8.21622312620400530036590680538211e-01
            -8.78693400099268773217886518978048e-01 -4.77386540051127017392218476743437e-01
            6.67588140216153647088503930717707e-01 -7.44530775081015172567333593178773e-01
            -4.66403903874177058752614755121613e-02 9.98911744842610804795413059764542e-01
            9.87930439740756671795907095656730e-01 -1.54898180214084690842213376527070e-01
            -4.07683614941688698962707348982804e-01 9.13123250228618843493677559308708e-01
            -9.87930439740756671795907095656730e-01 -1.54898180214084663086637760898157e-01
            4.66403903874177058752614755121613e-02 -9.98911744842610804795413059764542e-01
            1.70243555722398631768754739823635e-01 9.85402015288681321081298847275320e-01
            -8.93115016186791454977367266110377e-01 4.49828376007636032696268557629082e-01
            -3.20825769815367811688133770076092e-01 -9.47138229311000112708995857246919e-01
            8.93115016186791565999669728626031e-01 4.49828376007635810651663632597774e-01
            -5.95308864276666072079535751981894e-01 8.03496954638675653370682994136587e-01
            6.20009547460775078953076899779262e-01 7.84594265246365774046921615081374e-01
            -9.42034263469989197048448659188580e-01 -3.35516685797524916168299569108058e-01
            9.50596473416901455166794221440796e-01 2.95777266676101845133128875886541e-02
            4.81753674101715156918857019263669e-01 0.00000000000000000000000000000000e+00
            -7.07106781186547461715008466853760e-01 -0.00000000000000000000000000000000e+00
            -1.39515338943923122849710694026726e-01 -9.90219910019669469214420587377390e-01
            7.70513242775789253258267308410723e-01 0.00000000000000000000000000000000e+00
            1.85590558902697183585317475262855e-01 -3.83202491604425321547466865013121e-01
            -2.49475959426963295184620506006468e-01 4.12126616682904511890228604897857e-01
            -8.75534417063493108912552997935563e-01 -1.65324259210575696954492741497234e-01
            6.29407335104375631473772045865189e-01 -3.67752180525778238351364279878908e-01
            8.69969343870673927909820122295059e-01 -1.92473288722008084583592335548019e-01
            -5.87172973628806271761959578725509e-01 -3.93989719459664544665145058388589e-01
            -4.12602097386282840041360486793565e-01 3.81703069007967965564631640518201e-01
            4.10571428305382624124320045666536e-01 -4.87586346164877504349988157628104e-01
            -9.39577682361783184106229782628361e-01 1.47316923654866260839924052561400e-01
            9.19353038982236192566688259830698e-01 -3.93433589966752705890939978417009e-01
            2.62157881733017898706616932713587e-02 -5.61471687717300382836072003556183e-01
            -7.26750241965370413765867851907387e-01 -3.94837133696432196483527832242544e-01
            -1.02845977502097063460695380854304e-02 6.61231888422635250002201701136073e-01
            4.83154349693173335023033132529235e-01 5.73784359454594161675800023658667e-01
            -6.86149770525401825338462913350668e-01 4.92024530069868293313817275702604e-01
            7.42507536053895744743158502387814e-01 4.65398706909109116658385119080776e-01
            -3.66366063854022649959318869150593e-01 -7.00086316705688549433261869126000e-01
            -5.40090150678033564801694410562050e-01 6.02338049875692949086669614189304e-01
            7.82921552249708274651140982314246e-01 -4.25354521745265723442486205385649e-01
            -8.65001594258010486626631063700188e-01 -3.70174098660339401245522594763315e-01
            2.47200504266477855130901275515498e-01 -7.29782548423337673604294195683906e-01
            6.19786114784536157351624297007220e-01 5.73371448280105777683957057888620e-01
            -1.82957133172802599574424675665796e-01 -4.04777034558985102097317110292352e-02
            2.35597651203856894852606274071150e-01 7.73952352496392048664120011380874e-01
            -3.52637902129816493967950918886345e-01 7.28117440505963720021043172891950e-01
            6.24442087874597318020164493646007e-01 -6.14805234509524956010295682062861e-01
            -5.85011946789474879260239958966849e-01 -6.52437291701728927506565014482476e-01
            8.51016801698623037708557603764348e-02 3.58152747192536258236117419073707e-01
            -8.82848918242549851598255372664426e-01 3.77811792178284144849698122925474e-01
            9.12438524803243833360966164036654e-01 3.24975812141168385327461010092520e-01
            -1.62456334530964335849390067778586e-01 7.92537971682264030803821697190870e-01
            4.59486788675915747148792434018105e-01 -7.08351280661530413595983191044070e-01
            -7.38374233419929582034058057615766e-01 -5.65055577029584754633617649233202e-01
            -1.17796696706233078311143458449806e-01 -8.36070322417699296480009252263699e-01
            2.41029989751255113716865707829129e-01 6.12470735511341859980127821927454e-02
            -4.53770896495553521798882457005675e-01 -1.41190420137143352735664691977036e-02
            4.92410666622815240689448046396137e-01 7.59107192872271241412818199023604e-01
            -7.25849875152163481750733353692340e-01 2.58520049579236577130814112024382e-01
            6.12610579657759224403434927808121e-01 -1.90613249528190156767415430749679e-02
            7.52237928764343588561303022288484e-01 -2.41832673786562230011654150985123e-01
            -8.26680500754832570109442713146564e-01 2.57220919459963601183538628447423e-02
            6.55906416265418512612228596481145e-02 -8.41776403514350457513160108646844e-01
            4.15568921258815324182300798838696e-02 8.90036881743152541446306713623926e-01
            5.02005105108732196939058667339850e-01 2.52841053040110286254105176340090e-01
            -5.76470298255244428808907741768053e-01 7.29497298788840953243095555080799e-01
            -1.79701076101929646533506002015201e-01 2.13409373029398802268374879531621e-01
            6.77707622509918450326438232877990e-01 6.67248735914446999828442130819894e-01
            -7.36932900907815047197857438732171e-01 6.01197469012525131937252353964141e-01
            2.67263649213789811209807112390990e-01 -8.77977046837371544185657512571197e-01
            -3.74153523428301826303510324578383e-01 -8.38023086716906173698760085244430e-01
            -2.40056290007310893508396532070037e-02 -3.08083158560810255188755490962649e-01
            8.20694755398777564536771933489945e-01 -5.14405927729135292736373230582103e-01
            4.63663531985327448392553151279571e-01 -8.86011359468314774190389471186791e-01
            -5.89664120277779479906143933476415e-01 -7.46193488610364497048976772930473e-01
            -3.52467483963694316884129875688814e-01 8.60378397252632542979711161024170e-01
            2.70764591773613005898369010537863e-01 8.89477852198903518043948679405730e-01
            -7.46780663768384256329113668471109e-01 -1.89761159130401813044386472029146e-01
            -1.02228718520951153414522138973553e-01 -9.35310595736763117358236740983557e-01
            3.36509247777812936508468055762933e-01 -2.10921722910814096740850231981312e-01
            -3.80269878878140454858680641336832e-01 1.91527607262398397169889108226926e-01
            5.53713847221861565905953739274992e-01 -1.97211759809832659628980877641879e-01
            -5.89047158431277750878507504239678e-01 -1.69347280939523531229085051563743e-01
            6.63005986219993403985029090108583e-01 -6.94684838604860632749193882773397e-01
            -1.03334335348269412691024626838043e-01 9.45426100933121627001298747927649e-01
            -3.79464151399993732116655564823304e-01 -5.12168234576706571381521371222334e-01
            4.71498515663434980549340025390848e-01 8.36572239844345477699505408963887e-01
            2.32345956671816705751965059789654e-01 -5.67159953483874823554344857257092e-01
            -5.92402124006527253108345121290768e-01 4.24799933188803380268439013889292e-01
            1.40508332971050309279092971337377e-01 -1.66864861887356608072252583951922e-01
            -2.50695659466247167124208772293059e-01 6.11951852307541788178468777914532e-01
            2.50695659466247111613057541035232e-01 6.11951852307541788178468777914532e-01
            4.65076024826899026010096349637024e-01 -5.88532843529876115162835503724637e-01
            -7.68286721159375241008149259869242e-02 -7.02920589498487902169188146217493e-01
            3.07732865629280982755489048940944e-01 2.51051648963408591264112601493252e-01
            -3.39126582952540278625974679016508e-01 -9.07274214753657215482007813989185e-01
            1.35132208020237343237823779418250e-01 -9.59110330659360021954285002721008e-01
            -4.25386910258038097865096460736822e-01 8.78327673943004927359368139150320e-01
            1.35132208020237482015701857562817e-01 9.59110330659360132976587465236662e-01
            ];
        
    case   15
        
        % DEGREE.             :          15
        % CARDIN.             :         136
        % LEBESGUE CONST.     : 5.67314e+01
        % ABS.DET.VAND.MATRIX : 2.44186e+116
        % CONDITIONING VAND.  : 7.47865e+02
        % POINTS OUT DISK     :           0
        
        pts=[
            9.99516282291988078512190440960694e-01 3.10998622698369156513997779711644e-02
            -1.00000000000000000000000000000000e+00 -0.00000000000000000000000000000000e+00
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            -5.77379444863542756749552609107923e-01 2.05640543343572512746675329253776e-01
            6.58418280570842839871659180062125e-01 2.57847567004490074715761238621781e-01
            -8.39358674760744216669650086259935e-01 2.98947556019653504222333140205592e-01
            8.69602986211723494314185245457338e-01 2.50005282436187581396325185778551e-01
            3.05438557953331024652499081639689e-01 4.40247250454620375759162698159344e-01
            -3.21613039335863215573141360437148e-01 -3.58680436377076972132016408068012e-01
            -9.43618424191435267722738444717834e-01 -1.78180336394002186706231327661953e-01
            9.52031820332836509201968056004262e-01 -2.41916630304473939672149640500720e-01
            4.79169353634575578837484499672428e-01 -8.50182484394046311670933846471598e-01
            -7.12583964147506909725393597909715e-01 7.01586839984776666767629649257287e-01
            -1.56040645903459479493236017333402e-01 -5.12602839480523320148108723515179e-01
            7.74474761895350805041005060047610e-01 6.31824506806498931155147147364914e-01
            -9.80053383657033649889456228265772e-01 1.22570891038516954774095779612253e-01
            2.00807072855718332649033186498855e-01 -9.79630807749081733426521623187000e-01
            -7.54822939153257044253564345126506e-01 -6.55928601699939939351224893471226e-01
            7.94141650844753299587353012611857e-01 -6.07732703080531955208698491333053e-01
            -4.07683614941688698962707348982804e-01 9.13123250228618843493677559308708e-01
            9.61070940398724538056285382481292e-01 2.76301732750830253415585957554867e-01
            -2.00807072855718332649033186498855e-01 9.79630807749081733426521623187000e-01
            5.17849624898325311406210857967380e-01 8.55471662881163830149944260483608e-01
            -9.82635248222263668615994447463891e-01 -1.85547753829493261568117645765597e-01
            -8.47314704957777498073312472115504e-01 -5.31091132257275000405627451982582e-01
            8.78693400099268773217886518978048e-01 -4.77386540051127017392218476743437e-01
            -5.44204182756027332601433954550885e-01 -8.38952804078301195289668612531386e-01
            6.20009547460774967930774437263608e-01 -7.84594265246366107113829002628336e-01
            -9.19353038982236192566688259830698e-01 3.93433589966752705890939978417009e-01
            4.66403903874176087307468208109640e-02 9.98911744842610804795413059764542e-01
            9.92269872363276461690873020415893e-01 -1.24098752613259658006406027652702e-01
            1.55518119203506659242730592040971e-02 -9.99879063260149303360435624199454e-01
            -5.44204182756027110556829029519577e-01 8.38952804078301306311971075047040e-01
            6.44110411450397823962532584118890e-01 7.64932531574647578409553716483060e-01
            3.20825769815367811688133770076092e-01 9.47138229311000112708995857246919e-01
            -8.30387982764797305534898441692349e-01 5.57185604695428882138230619602837e-01
            8.78693400099268773217886518978048e-01 4.77386540051127017392218476743437e-01
            -3.20825769815367811688133770076092e-01 -9.47138229311000112708995857246919e-01
            -9.31144062097659541699101737322053e-01 -3.64651526282655180644098891207250e-01
            5.09041415750371206705437998607522e-01 0.00000000000000000000000000000000e+00
            9.50596473416901455166794221440796e-01 2.95777266676101845133128875886541e-02
            -7.07106781186547461715008466853760e-01 -0.00000000000000000000000000000000e+00
            -1.70243555722398631768754739823635e-01 -9.85402015288681321081298847275320e-01
            7.70513242775789253258267308410723e-01 0.00000000000000000000000000000000e+00
            -9.76389420563607002279127300425898e-01 2.16017821976483392898416013849783e-01
            1.97887384972315449882529492242611e-01 -4.08592653779509373812572903261753e-01
            -2.47063528889760436468847615287814e-01 3.80876602781402173736324812125531e-01
            6.47662178602191218601547006983310e-01 -3.78418180311696217277273035506369e-01
            -8.11459033367125659452767649781890e-01 -2.33289268826120610444974090569303e-01
            8.63562631523739021588426112430170e-01 -2.19436112757292800523600817541592e-01
            -5.74635921652868808529035504761850e-01 -4.12060138264013553488496199861402e-01
            3.75240196866717501400501078023808e-01 -4.18488372972401989624557927527349e-01
            -9.18554492564195745885058386193123e-01 1.44020685669936754358388952823589e-01
            -4.05818705979538674188944469278795e-01 4.25208984723534688665580461019999e-01
            7.56328338316431980636878051882377e-01 -4.10906658148006420638864710781490e-01
            -4.36647992887536431916295498467662e-02 5.60384786517919342152538320078747e-01
            -7.14119319527053653118287002143916e-01 4.17247976412776333532406169979367e-01
            9.52013107532729829785012043430470e-01 -3.06057254587887683783264947123826e-01
            3.08438435665265194007833571276933e-02 -6.60592189275570618178790027741343e-01
            4.51967508800651784905966223959695e-01 5.71944604619354368679751132731326e-01
            -7.15411467084352103995570359984413e-01 -4.48415073951301201837793541926658e-01
            7.56622207695807680316590904112672e-01 4.42081703207699294733146189173567e-01
            -3.87961434127866156007513609438320e-01 -6.88353738430958816962856872123666e-01
            1.64659522627356807999987609036907e-01 1.43086391299526716203160958684748e-01
            -8.65001594258010486626631063700188e-01 -3.70174098660339401245522594763315e-01
            8.76306680043863583939867112349020e-01 0.00000000000000000000000000000000e+00
            -5.52149182388863013493107700924156e-01 6.15786941019090705573546529194573e-01
            3.38574066714893706642186543831485e-01 1.05347026651995672591466401968319e-02
            6.31835025983950360206620189273963e-01 5.84518038208162060698214190779254e-01
            -9.50596473416901455166794221440796e-01 -2.95777266676101845133128875886541e-02
            -2.59553500014057525469013398833340e-01 7.66250923534795069791414334758883e-01
            1.58668715137436550088878561837191e-01 7.74060193020583153789004882128211e-01
            2.11413876881949236441471384750912e-01 -7.80905032541869048579030732071260e-01
            -5.64438256239785052237323270674096e-01 -6.70315487201727289168218248960329e-01
            4.71462607478480622003047528778552e-01 -6.79547854158874531371736793516902e-01
            -8.20687135869564876600179559318349e-01 1.02639667578131574265931647005345e-01
            -1.66083629135901550410991944772832e-01 -8.10233611050163227140785693336511e-01
            8.49402855945499579704005554958712e-01 -4.27812208216528766513420123374090e-01
            6.64196014914965049413808628742117e-01 -6.14455570928789107654210965847597e-01
            1.31308290965072935879698334815657e-02 8.44225815235340126818641692807432e-01
            7.77522723582480423942797642666847e-01 2.23532797370666669323213682218920e-01
            -4.06091666456825928221974209009204e-01 7.20521918318004495063178183045238e-01
            -8.93906187832087195666019852069439e-01 -1.40156063835460503730345749318076e-01
            -4.53770896495553632821184919521329e-01 1.41190420137142658846274301254198e-02
            -2.12992753067037021708074462367222e-01 -4.71228278853713516971524200016574e-02
            -7.20443918107661396277308085700497e-01 5.87745586603921066348732438200386e-01
            3.81968427637002616581440861409646e-01 7.88678336064846052799737208260922e-01
            2.85973315340901923509164817005512e-02 -3.67012096419754851872596645989688e-01
            8.57652710627436754009522701380774e-01 4.31967349118446364286683092359453e-01
            -8.57652710627436642987220238865120e-01 4.31967349118446586331288017390762e-01
            9.45714351462050850649632138811285e-01 2.09231224870104259849767913692631e-01
            6.92168147585572879343374097516062e-02 -8.88313941521192140804430437128758e-01
            1.40023288572894616388353483671381e-01 5.17207727005983053381044101115549e-01
            4.76261511466449072305096024138038e-01 2.98517497566481826432038815255510e-01
            -7.10198387414805010209306601609569e-01 -6.17150607146509377898269121942576e-01
            7.17879274930765309825630993145751e-01 6.23825170871096235103436811186839e-01
            -5.30002589884955188637150058639236e-01 7.63925106555574551236986735602841e-01
            5.60114661985896211682245393603807e-01 7.55994832532815519421376393438550e-01
            -3.47910136167665140494165143536520e-01 -8.49253843156532628455579470028169e-01
            -8.57531617761826514545475674822228e-01 -4.65889526359837990998613577176002e-01
            1.64659522627356502688655837118858e-01 -1.43086391299526494158556033653440e-01
            3.21330168319705899993010689286166e-01 -8.59663001351095412694292008382035e-01
            4.35884418475371071544088863447541e-01 -9.00002652068530029083603949402459e-01
            -5.17568934205198449838292162894504e-01 -7.97891531182759572438101258740062e-01
            -2.98296256841325657660490833222866e-01 8.80626854499202527115642169519560e-01
            7.40517886824198390272044889570680e-01 -2.12894144086411857275109582587902e-01
            -5.26081519010394779711781154674100e-02 3.04505969015524291521046507114079e-01
            -1.29718081570789151024669649814314e-01 -9.20683188194668766790584868431324e-01
            1.58288454984692517069078121494385e-01 9.16203622962311170496718659705948e-01
            -3.23468614858522185429023920733016e-01 1.75737706514007302471824800704781e-01
            5.16753061589756867988398880697787e-01 -2.21142481209002511599592821767146e-01
            -4.92613992332052441636136563829496e-01 -2.10812259548965735822534384169558e-01
            -7.30912379726026961712648244429147e-02 9.38037468505137583818509483535308e-01
            7.31108788497465744704584267310565e-01 -6.35321398509210655269896506069927e-01
            -6.90196828563563347103126943693496e-01 6.79545199278702050094125297619030e-01
            -1.43279911055378017525896439110511e-01 -2.73792567284959487139417433354538e-01
            3.64036200521267283569670780707384e-01 8.88617807305844609189193761267234e-01
            1.22883421403890934209268692711703e-02 -7.90059453604517236691151538252598e-01
            2.77843185093858524581378333095927e-01 -5.73683281265924271963285718811676e-01
            6.26163684451213065074171026935801e-01 4.49009685382558310084988306698506e-01
            -7.32400512634831746261454554769443e-01 -1.62037359502022221269257329367974e-01
            1.07203285673854495363421790443681e-01 3.52169194890785797191767869662726e-01
            -2.88255537847462939282650040695444e-01 5.95182432635673275278520577558083e-01
            3.36509247777812769975014361989452e-01 2.10921722910814041229699000723485e-01
            -5.14385502756606771690428558940766e-01 -5.73670821854044721277432472561486e-01
            5.00765253930148146466194702952635e-01 -5.58480776068815298884828735026531e-01
            -1.09967916686177433205351405831607e-02 7.07021265997704762007458612060873e-01
            -1.78887166251082835310981522525253e-01 -6.60760260598351134575523246894591e-01
            3.67627081832808677575030742445961e-01 -2.14798356205881169067595237720525e-01
            1.99350188459021443465601919342589e-01 6.54877272927298914595439782715403e-01
            1.05238642993211214959892174647393e-01 -9.62848985066131146659529349562945e-01
            -4.52497012764203621060232762829401e-01 8.64673336973265316274250835704152e-01
            2.82065666536236270278692472857074e-01 -9.26602557617558164615445548406569e-01
            -3.13099246424719646064716016553575e-01 -9.24328073857589971318304833403090e-01
            4.50100707996473670124970567485434e-01 6.93881566986335296398635819059564e-01
            1.06035454489099068009494430953055e-01 9.70139169719633653876655898784520e-01
            ];
        
    case   16
        
        % DEGREE.             :          16
        % CARDIN.             :         153
        % LEBESGUE CONST.     : 5.64720e+01
        % ABS.DET.VAND.MATRIX : 1.14228e+135
        % CONDITIONING VAND.  : 5.60368e+02
        % POINTS OUT DISK     :           0
        
        pts=[
            9.99516282291988078512190440960694e-01 3.10998622698369156513997779711644e-02
            -1.00000000000000000000000000000000e+00 -0.00000000000000000000000000000000e+00
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            -5.77379444863542756749552609107923e-01 2.05640543343572512746675329253776e-01
            6.58418280570842839871659180062125e-01 2.57847567004490074715761238621781e-01
            -8.39358674760744216669650086259935e-01 2.98947556019653504222333140205592e-01
            8.69602986211723494314185245457338e-01 2.50005282436187581396325185778551e-01
            3.05438557953331024652499081639689e-01 4.40247250454620375759162698159344e-01
            -3.21613039335863215573141360437148e-01 -3.58680436377076972132016408068012e-01
            -9.43618424191435267722738444717834e-01 -1.78180336394002186706231327661953e-01
            9.52031820332836509201968056004262e-01 -2.41916630304473939672149640500720e-01
            4.79169353634575578837484499672428e-01 -8.50182484394046311670933846471598e-01
            -7.12583964147506909725393597909715e-01 7.01586839984776666767629649257287e-01
            -1.56040645903459479493236017333402e-01 -5.12602839480523320148108723515179e-01
            7.74474761895350694018702597531956e-01 -6.31824506806499042177449609880568e-01
            -9.80053383657033649889456228265772e-01 1.22570891038516954774095779612253e-01
            1.61407992027690960767927208507899e-01 -3.93999321363938448392616464843741e-01
            5.17849624898325311406210857967380e-01 8.55471662881163830149944260483608e-01
            -7.54822939153257044253564345126506e-01 -6.55928601699939939351224893471226e-01
            7.74857107102889908212262071174337e-01 6.32136427974326342571487202803837e-01
            -2.31176322114969662324668320252385e-01 9.72911870671437473134801621199585e-01
            -9.19353038982236192566688259830698e-01 3.93433589966752705890939978417009e-01
            9.69198999199666122450480543193407e-01 2.46278906832001348581329125408956e-01
            -4.35884418475371071544088863447541e-01 9.00002652068530029083603949402459e-01
            1.39515338943923122849710694026726e-01 9.90219910019669469214420587377390e-01
            -8.63421704896663477768470329465345e-01 -5.04482863448639795578287703392562e-01
            8.78693400099268773217886518978048e-01 -4.77386540051127017392218476743437e-01
            3.20825769815367700665831307560438e-01 -9.47138229311000112708995857246919e-01
            -6.20009547460775078953076899779262e-01 -7.84594265246365774046921615081374e-01
            6.44110411450397712940230121603236e-01 -7.64932531574647689431856178998714e-01
            -9.87930439740756671795907095656730e-01 -1.54898180214084663086637760898157e-01
            9.92269872363276461690873020415893e-01 -1.24098752613259658006406027652702e-01
            -4.66403903874177058752614755121613e-02 9.98911744842610804795413059764542e-01
            1.55518119203506659242730592040971e-02 -9.99879063260149303360435624199454e-01
            -4.63663531985327337370250688763917e-01 -8.86011359468314774190389471186791e-01
            6.44110411450397823962532584118890e-01 7.64932531574647578409553716483060e-01
            -8.47314704957777609095614934631158e-01 5.31091132257275000405627451982582e-01
            8.93115016186791565999669728626031e-01 4.49828376007635810651663632597774e-01
            -5.70032258213782561107052515581017e-01 8.21622312620400530036590680538211e-01
            3.20825769815367811688133770076092e-01 9.47138229311000112708995857246919e-01
            -9.42034263469989197048448659188580e-01 -3.35516685797524916168299569108058e-01
            9.50596473416901455166794221440796e-01 2.95777266676101845133128875886541e-02
            5.08795183403461415139190648915246e-01 -1.58311179194792553404091961510858e-02
            -7.07106781186547461715008466853760e-01 -0.00000000000000000000000000000000e+00
            -1.39515338943923122849710694026726e-01 -9.90219910019669469214420587377390e-01
            -2.91214712227252259246768062439514e-01 -9.56657719031419606636745811556466e-01
            7.90155012375690413151119173562620e-01 0.00000000000000000000000000000000e+00
            -9.76389420563607002279127300425898e-01 2.16017821976483392898416013849783e-01
            -2.31710871400610557424570856710488e-01 3.57208730576990318716923411557218e-01
            3.62734749950658741379783123193192e-01 -3.57136758297679934859303330085822e-01
            6.40539921794616939898503460426582e-01 -3.47999810850526669625537579122465e-01
            -8.19437691912045407249820527795237e-01 -2.63436341693240150263477517000865e-01
            -5.87172973628806271761959578725509e-01 -3.93989719459664544665145058388589e-01
            8.89114955298166087516165134729818e-01 -1.67888627189231087744047954402049e-01
            -4.04454356263115710756750331711373e-01 3.51464120383933653624097814827110e-01
            1.40293757567952454001769524438714e-01 4.60873371001152720705107412868529e-01
            -9.18554492564195745885058386193123e-01 1.44020685669936754358388952823589e-01
            7.65528996461353306024477660685079e-01 -3.56167097525037057437202747678384e-01
            -4.36647992887536431916295498467662e-02 5.60384786517919342152538320078747e-01
            9.52013107532729829785012043430470e-01 -3.06057254587887683783264947123826e-01
            -7.26750241965370413765867851907387e-01 -3.94837133696432196483527832242544e-01
            4.87797290388368331992552384690498e-01 4.80269240869232894031171099413768e-01
            1.02845977502097028766225861318162e-02 -6.61231888422635027957596776104765e-01
            5.00765253930148146466194702952635e-01 -5.58480776068815298884828735026531e-01
            -6.72133574057836469251014932524413e-01 4.81973790746527797068665677215904e-01
            7.70004896217444501083093655324774e-01 4.18337014009830032712500269553857e-01
            -4.09181476770282881449958267694456e-01 -6.75955222370918429319885945005808e-01
            2.47200504266477855130901275515498e-01 -7.29782548423337673604294195683906e-01
            -8.26680500754832570109442713146564e-01 2.57220919459963601183538628447423e-02
            -5.08947070130900014461872160609346e-01 6.04415273952933929102471211081138e-01
            9.52874801707685237239786602003733e-01 2.10815413344350582880437627864012e-01
            6.31835025983950360206620189273963e-01 5.84518038208162060698214190779254e-01
            -9.14212175845365071502612863696413e-01 2.93904849036369153481018656748347e-01
            -1.50344618426986015302659893677628e-01 -4.32231137525649553032813798836287e-02
            8.65001594258010375604328601184534e-01 -3.70174098660339734312429982310277e-01
            2.35597651203856894852606274071150e-01 7.73952352496392048664120011380874e-01
            -3.22133251809616216743137329103774e-01 7.21508913084924929925989545154152e-01
            6.34916961087438669331106666504638e-01 -6.25118451711136446924399479030399e-01
            -5.54412901166159599597449414432049e-01 -6.58409577748820362685933105240110e-01
            -8.86341922194870845963521333032986e-01 -3.15681197330148488600087830491248e-01
            3.91484068092108616898627815317013e-01 7.48084133111102334012798564799596e-01
            -2.53502090118902345672324827319244e-01 -7.48386019302722749202416707703378e-01
            8.62296885495731801896113211114425e-01 4.01188956044199296346874916707748e-01
            3.85753608660151098286306137197244e-02 8.26180499574031168563692517636809e-01
            -7.38374233419929582034058057615766e-01 -5.65055577029584754633617649233202e-01
            7.55590219348507785568358485761564e-01 -5.41818912678008524075323748547817e-01
            -1.15390426653849978433363787644339e-01 8.18991651837173151307069929316640e-01
            -9.17380444830022401259128628225881e-02 -8.39329361441008225952487009635661e-01
            2.76834469180035236401948850470944e-01 -3.46224482496619925808722939564177e-02
            -4.73388141132926654197632387877093e-01 -8.93883121286790233162022900614829e-02
            4.45734935818216548852888081455603e-01 -7.36340413152767436777423881721916e-01
            -7.47192607105389483557189578277757e-01 5.71804012993040844392567123577464e-01
            7.47192607105389705601794503309065e-01 5.71804012993040622347962198546156e-01
            6.68657521810039345844245417538332e-02 -8.58140902320898790733849637035746e-01
            -3.38574066714893706642186543831485e-01 1.05347026651995117479954089390048e-02
            6.60991977063876245956919319723966e-01 -2.05667079289744009207296926433628e-02
            -9.15979318505400708083641347911907e-01 -5.70564723298372170057213281779696e-02
            8.75882794985077262062134195730323e-01 2.72530170555022044354132049193140e-02
            -4.80577417671127632425864817378169e-03 -3.08979622867089021021058670157799e-01
            -6.20707755030848629118622739042621e-01 6.92247207690482424524702764756512e-01
            4.54632532267640054879365152373794e-01 2.28981273367617255143358079294558e-01
            -3.74153523428301826303510324578383e-01 -8.38023086716906173698760085244430e-01
            5.83355059773855488813865122210700e-01 7.38209655602076408875689139676979e-01
            2.67263649213789811209807112390990e-01 -8.77977046837371544185657512571197e-01
            -7.46780663768384256329113668471109e-01 1.89761159130402201622445090833935e-01
            7.84098461283720959968945862783585e-01 1.99243820983173447469738448489807e-01
            -4.81484404456514192283123065863037e-01 7.95397436490627440797140934591880e-01
            -1.22105262595867028463381132041832e-01 2.16649405317583421126670373269008e-01
            1.58288454984692517069078121494385e-01 9.16203622962311170496718659705948e-01
            -2.98296256841325657660490833222866e-01 8.80626854499202527115642169519560e-01
            5.32928783629538327382135776133509e-01 -2.47948410746402780890917938450002e-01
            -5.12031249939596810882846966705984e-01 -7.89354559417495660333941032149596e-01
            6.63005986219993403985029090108583e-01 -6.94684838604860632749193882773397e-01
            -4.55449580189087932513558598657255e-01 -3.71560469800152115560365473356796e-01
            -7.14115570393593745990301613346674e-01 -2.29576934607081978301224012284365e-01
            8.57531617761826514545475674822228e-01 -4.65889526359837990998613577176002e-01
            -8.42627714430384422428232937818393e-01 -4.92333282550383777298463883198565e-01
            1.66085878479504622795559498626972e-01 2.24168504073796737152690639049979e-01
            -1.88935513119935755144140898664773e-01 -9.21715787686204879314288973546354e-01
            -5.56302623021174391659826596878702e-01 3.98913689731381027048229270803859e-01
            4.90994080973322089000987489271211e-01 -8.71162908099950561435775853169616e-01
            3.01858797004214585157910732959863e-01 8.91144145500077256016879800881725e-01
            3.59888683215321747432824395218631e-01 -5.54809443783529587790326331742108e-01
            7.27006897974432297893088161799824e-01 -1.84736534231173171738404903408082e-01
            -2.49872163256504992157047695400252e-01 5.59659680919652346986481461499352e-01
            -7.30912379726026961712648244429147e-02 9.38037468505137583818509483535308e-01
            -6.68729206031876799443125491961837e-01 -7.00681517539821552276180227636360e-01
            -3.30089773992552959480661911584320e-01 -5.45298160470657489184986843611114e-01
            3.17846728893326380127604124936624e-01 1.85712528444116065085012223789818e-01
            -8.10389551292669430004877995088464e-01 5.43766771133245896407970576547086e-01
            7.38817291750868548882991149184818e-02 -9.48182465183559863852735816180939e-01
            6.39824937362150292230467130139004e-01 4.29318887101863666533319019436021e-01
            1.32796099920397786542380913488159e-01 6.47841476801061078027998973993817e-01
            1.66572862903142326906902326300042e-01 -6.15274592398701858364518102462171e-01
            3.17399528945229825183105276664719e-01 6.06516511943977931764493405353278e-01
            -3.99975241484437568639798143976805e-02 3.65945193097667265114125711988891e-01
            -5.56304035636626781347047199233202e-01 8.01834986869126797870421796687879e-01
            -3.33768445739763108814912584421108e-01 -1.55287832484573368185820640974271e-01
            2.37418903539611189645874844700302e-02 -5.08487448804356256815140113758389e-01
            5.01580426673295831641041786497226e-01 8.28595447489404279828306698618690e-01
            -5.80020025876412326226727600442246e-01 -5.36583370190274799860219445690745e-01
            4.81142249428901436214545128677855e-01 -4.18104626285854985567880248709116e-01
            2.82065666536236270278692472857074e-01 -9.26602557617558164615445548406569e-01
            -1.20380372704617863433007585172163e-01 6.96784447205516577206196870974964e-01
            1.92621045170938948176697635972232e-01 -2.01824602579188827622402868655627e-01
            -3.67178644538921683526666583929909e-01 -8.96288560128093769741042251553154e-01
            5.73907301901681488054407509480370e-01 1.26972089990117942281955265571014e-01
            1.66143539641541992946116579332738e-01 9.61670343968445973104053337010555e-01
            -3.48971230364127860124057178836665e-01 1.89593057398741271502018435057835e-01
            5.98564330859512952476286784531112e-02 -7.68184785524348434648800321156159e-01
            -1.36155357822940165091196718094579e-01 9.66372208193673642995236150454730e-01
            -3.32197700612490320892788986384403e-01 8.10899553087355284830550772312563e-01
            2.66319595659801379439191038045465e-02 -1.54150837111250216260316392435925e-01
            ];
        
    case   17
        
        % DEGREE.             :          17
        % CARDIN.             :         171
        % LEBESGUE CONST.     : 5.80052e+01
        % ABS.DET.VAND.MATRIX : 1.03123e+154
        % CONDITIONING VAND.  : 6.87427e+02
        % POINTS OUT DISK     :           0
        
        pts=[
            9.99516282291988078512190440960694e-01 3.10998622698369156513997779711644e-02
            -1.00000000000000000000000000000000e+00 -0.00000000000000000000000000000000e+00
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            -5.77379444863542756749552609107923e-01 2.05640543343572512746675329253776e-01
            6.58418280570842839871659180062125e-01 2.57847567004490074715761238621781e-01
            -8.39358674760744216669650086259935e-01 2.98947556019653504222333140205592e-01
            8.69602986211723494314185245457338e-01 2.50005282436187581396325185778551e-01
            3.05438557953331024652499081639689e-01 4.40247250454620375759162698159344e-01
            -3.21613039335863215573141360437148e-01 -3.58680436377076972132016408068012e-01
            -9.43618424191435267722738444717834e-01 -1.78180336394002186706231327661953e-01
            9.52031820332836509201968056004262e-01 -2.41916630304473939672149640500720e-01
            4.79169353634575578837484499672428e-01 -8.50182484394046311670933846471598e-01
            -7.12583964147506909725393597909715e-01 7.01586839984776666767629649257287e-01
            -1.56040645903459479493236017333402e-01 -5.12602839480523320148108723515179e-01
            7.74474761895350694018702597531956e-01 -6.31824506806499042177449609880568e-01
            -9.80053383657033649889456228265772e-01 1.22570891038516954774095779612253e-01
            1.61407992027690960767927208507899e-01 -3.93999321363938448392616464843741e-01
            9.82635248222263668615994447463891e-01 -1.85547753829493428101571339539078e-01
            -4.63663531985327337370250688763917e-01 -8.86011359468314774190389471186791e-01
            -6.90420020571746673176960484852316e-01 -7.23408733147249849437798729923088e-01
            7.54822939153257044253564345126506e-01 6.55928601699939939351224893471226e-01
            -9.19353038982236192566688259830698e-01 3.93433589966752705890939978417009e-01
            4.90994080973322200023289951786865e-01 8.71162908099950450413473390653962e-01
            9.69198999199666122450480543193407e-01 2.46278906832001348581329125408956e-01
            -4.35884418475371071544088863447541e-01 9.00002652068530029083603949402459e-01
            -1.70243555722398520746452277307981e-01 9.85402015288681321081298847275320e-01
            -8.47314704957777498073312472115504e-01 -5.31091132257275000405627451982582e-01
            8.63421704896663477768470329465345e-01 -5.04482863448639795578287703392562e-01
            3.20825769815367700665831307560438e-01 -9.47138229311000112708995857246919e-01
            -9.82635248222263668615994447463891e-01 -1.85547753829493261568117645765597e-01
            6.20009547460774967930774437263608e-01 -7.84594265246366107113829002628336e-01
            1.08652150085474491492121273950033e-01 9.94079830940052588950095469044754e-01
            9.31144062097659541699101737322053e-01 3.64651526282655180644098891207250e-01
            -4.66403903874177058752614755121613e-02 9.98911744842610804795413059764542e-01
            -7.74857107102889908212262071174337e-01 -6.32136427974326342571487202803837e-01
            1.55518119203506659242730592040971e-02 -9.99879063260149303360435624199454e-01
            -5.95308864276666072079535751981894e-01 8.03496954638675653370682994136587e-01
            6.20009547460775078953076899779262e-01 7.84594265246365774046921615081374e-01
            -8.47314704957777609095614934631158e-01 5.31091132257275000405627451982582e-01
            3.50126449191390798354461821872974e-01 9.36702444523675015197738957795082e-01
            8.47314704957777498073312472115504e-01 5.31091132257275000405627451982582e-01
            -2.91214712227252259246768062439514e-01 -9.56657719031419606636745811556466e-01
            -9.42034263469989197048448659188580e-01 -3.35516685797524916168299569108058e-01
            9.59829174616289138022295901464531e-01 2.98650013631469937736007125295146e-02
            4.81753674101715156918857019263669e-01 0.00000000000000000000000000000000e+00
            -7.07106781186547461715008466853760e-01 -0.00000000000000000000000000000000e+00
            -1.08652150085474491492121273950033e-01 -9.94079830940052588950095469044754e-01
            7.70513242775789253258267308410723e-01 0.00000000000000000000000000000000e+00
            -9.69198999199666011428178080677753e-01 2.46278906832001820426114591100486e-01
            1.70243555722398520746452277307981e-01 -9.85402015288681321081298847275320e-01
            -4.53990499739546748969587497413158e-01 -0.00000000000000000000000000000000e+00
            9.17754625683981140049638725031400e-01 0.00000000000000000000000000000000e+00
            -2.49475959426963295184620506006468e-01 4.12126616682904511890228604897857e-01
            -5.74635921652868808529035504761850e-01 -4.12060138264013553488496199861402e-01
            5.99593349122257146532888327783439e-01 -2.78964511911824863954478814775939e-01
            3.57711613538010786417231656741933e-01 -3.98939538974888763256387846922735e-01
            -8.27234149333384571889382641529664e-01 -2.37824513512651575242173862534401e-01
            8.49315557290494571596184414374875e-01 -2.15815851210783504754431305627804e-01
            -9.18554492564195745885058386193123e-01 1.44020685669936754358388952823589e-01
            7.26750241965370413765867851907387e-01 -3.94837133696432196483527832242544e-01
            1.40293757567952454001769524438714e-01 4.60873371001152720705107412868529e-01
            9.42034263469989086026146196672926e-01 -3.35516685797525027190602031623712e-01
            -7.14119319527053542095984539628262e-01 -4.17247976412776278021254938721540e-01
            -2.62157881733017898706616932713587e-02 5.61471687717300382836072003556183e-01
            -4.25957857670647421777232466411078e-01 5.05858959302373567545885180152254e-01
            5.03872953213493790514121428714134e-01 4.96096812144476817518778943849611e-01
            1.02845977502097028766225861318162e-02 -6.61231888422635027957596776104765e-01
            -4.46547768943064027347134015144547e-01 -6.02711960088833764714877361257095e-01
            5.00765253930148146466194702952635e-01 -5.58480776068815298884828735026531e-01
            -7.01119762849604311050200067256810e-01 4.70447365732077293909441095820512e-01
            7.56328338316431869614575589366723e-01 4.10906658148006420638864710781490e-01
            -8.95730824669896619205644583416870e-01 -2.87963385040670538650431353744352e-01
            2.46767482595108028631258889618039e-01 3.08621047846906887412998798936314e-02
            -1.56358794919347593443958999159804e-01 4.86509031700677399556731472785032e-03
            9.30715679095851955437979086127598e-01 2.36500079146191061196802252197813e-01
            -9.03076752422127704278409510152414e-01 -5.62527697900076298354044013194653e-02
            -5.71032987105135059202609681960894e-01 5.98317310446660699518872661428759e-01
            -8.26680500754832570109442713146564e-01 2.57220919459963601183538628447423e-02
            3.67412451494770531112976641452406e-01 2.28861699700323856232486718909058e-02
            6.31835025983950471228922651789617e-01 -5.84518038208162171720516653294908e-01
            -2.35597651203856617296850117782014e-01 7.73952352496392159686422473896528e-01
            2.35597651203856589541274502153101e-01 -7.73952352496392048664120011380874e-01
            6.31835025983950360206620189273963e-01 5.84518038208162060698214190779254e-01
            -2.83258247575991717326360230799764e-01 -7.57808196292209412803231316502206e-01
            -8.12376877634941041428362495935289e-01 -4.74658224485785706914242609855137e-01
            -5.85011946789474879260239958966849e-01 -6.52437291701728927506565014482476e-01
            8.26745721960402857320104885729961e-01 -4.49163814891701540510382528736955e-01
            4.22619240503915805096113444960793e-01 7.49846527368601578622531178552890e-01
            -9.12438524803243722338663701521000e-01 3.24975812141168496349763472608174e-01
            6.42508010276486435374820871402335e-02 8.24581173026430747086124029010534e-01
            7.38374233419929359989453132584458e-01 -5.65055577029585087700525036780164e-01
            -7.06630329000015655616095955338096e-01 2.51674780062448311124256861148751e-01
            9.43704728034454376484063686802983e-01 -1.18024927336940810285170755378203e-01
            -6.55906416265419622835253221637686e-02 -8.41776403514350457513160108646844e-01
            2.40858031426117058515501412330195e-01 7.91233015640698922332774145615986e-01
            -3.99094688368858585558740514898091e-01 7.62627213497277023357412417681189e-01
            6.12610579657759113381132465292467e-01 1.90613249528191128212561977761652e-02
            8.35687083989954060392335577489575e-01 4.54021579707221623056767612069962e-01
            -8.05844171502799855311138799152104e-01 5.05097682079852616787718488922110e-01
            -5.83241597269268496894767395133385e-01 -7.29434166139658335392681465236819e-02
            -4.46373821599886422273328889787081e-01 -3.41596450578711186008007416603505e-01
            -3.06628253566512409822308882212383e-01 3.83486235382296536400303921254817e-02
            5.38651564970349494565482473262819e-01 7.27025781131135917512153810093878e-01
            -7.20443918107661396277308085700497e-01 -5.87745586603920844304127513169078e-01
            6.68657521810039345844245417538332e-02 -8.58140902320898790733849637035746e-01
            -4.80577417671127632425864817378169e-03 -3.08979622867089021021058670157799e-01
            -4.00034941319343972931932285064249e-01 -8.25981597063743855358097789576277e-01
            4.37478929485101697416382648953004e-01 -7.76211834747967399650292463775259e-01
            -6.49602919856937188747281197720440e-01 6.80641365111786522668069210340036e-01
            2.59867316391299729705366416965262e-01 -5.82046665293017406739295438455883e-01
            7.36932900907815158220159901247825e-01 6.01197469012525020914949891448487e-01
            -9.08719097980839718431411711208057e-01 -3.55869536765790861476688178299810e-01
            7.65817447207118973118156191048911e-01 1.94598512675711543540302272958797e-01
            -1.54040974725632412445364138875448e-01 8.91618400987733039642080257181078e-01
            -5.30002589884955188637150058639236e-01 7.63925106555574551236986735602841e-01
            8.64556302852384495061244251701282e-01 -3.07921990384837418464059055622783e-01
            4.80577417671128152842907610420298e-03 3.08979622867089354087966057704762e-01
            3.29427242746408333040619709208841e-01 8.81325316284738091709982654720079e-01
            -4.22701874368687280369982772754156e-01 2.83630549049887770163280720225885e-01
            6.06031099216872704005254490766674e-01 -7.19710308506056883892654241208220e-01
            -6.86713424025658158988960622082232e-01 -2.44581137922802671624822323792614e-01
            1.37461900573393552837941911093367e-01 6.70603434323217451051846182963345e-01
            -2.70764591773613005898369010537863e-01 -8.89477852198903407021646216890076e-01
            5.09625598275586733088005075842375e-01 2.37106459717805806430845905197202e-01
            -5.66172374578590376081876911484869e-01 -7.64171014532423864551446968107484e-01
            7.14115570393593634967999150831020e-01 -2.29576934607082283612555784202414e-01
            2.04795800836131108857429694580787e-01 -1.89459011949498296045746315030556e-01
            2.45872772057113303523223635238537e-01 -9.08186765675896290339608185604447e-01
            -5.74635921652868697506733042246196e-01 4.12060138264013775533101124892710e-01
            4.71498515663434980549340025390848e-01 8.36572239844345477699505408963887e-01
            -3.29427242746407999973712321661878e-01 8.81325316284738202732285117235733e-01
            6.22883217959981694278326358471531e-01 -4.17951089920782503650542594186845e-01
            -7.93086479827163626232788828929188e-01 -5.68706744947136688672628679341869e-01
            -1.51828126858278983357664060349634e-01 2.34060370724518296281502216515946e-01
            7.75016148413102956560294387600152e-01 -5.93096531714634944698616436653538e-01
            -7.38817291750869797883893852485926e-02 -9.48182465183559863852735816180939e-01
            1.46324007582511426972526891177040e-02 9.40766981901440146884851856157184e-01
            4.90994080973322089000987489271211e-01 -8.71162908099950561435775853169616e-01
            3.22745376284550911094584080274217e-01 2.31434373252114994334860398339515e-01
            1.03334335348269412691024626838043e-01 -9.45426100933121627001298747927649e-01
            -8.15679528628174499438330258271890e-01 5.47316315781874784818228363292292e-01
            6.69509961182017110381536895147292e-01 4.19644320181366548005996719439281e-01
            1.27999245549447804304321607560269e-01 -6.24440177956151365457060364860808e-01
            -2.77843185093858580092529564353754e-01 -5.73683281265924271963285718811676e-01
            3.37996168496707827255676193090039e-01 6.45874484591396291754961112019373e-01
            -3.65119120180470435599318079766817e-01 -1.56251320360164924760937310566078e-01
            -2.23178998160118019500330888149620e-01 5.97076609395631674459536952781491e-01
            6.28598147073706536858139770629350e-01 7.46510479315938724198531417641789e-01
            -7.84047013277229476457819146162365e-01 -9.80572514069376760525997838158219e-02
            7.49214253156779003139931205623725e-03 -4.81695412382958598751514500690973e-01
            1.36982981709462159347623355643009e-01 2.43046703271156822045995227199455e-01
            -1.91593486840099885837318538506224e-01 -2.42452961650252041270192648880766e-01
            -5.66290875310813263121545446665550e-02 7.26765716176852816232667464646511e-01
            -5.56304035636626781347047199233202e-01 8.01834986869126797870421796687879e-01
            4.89903651646915017092709376811399e-01 6.19951091365638085051159578142688e-01
            3.80269878878140510369831872594659e-01 -1.91527607262398424925464723855839e-01
            -3.88444736040215177208523300578236e-01 6.41698764082337724801163858501241e-01
            -3.69958727564431211209949879048509e-01 -9.03074784351726056463860459189164e-01
            6.26355280389529367468526288575958e-01 -1.18272589534903424524081572144496e-01
            6.32701445229335690534355762792984e-01 -7.51383473433396509300052912294632e-01
            -6.24299389990488395874024263321189e-01 -5.42505804577096628626975416409550e-01
            -1.21524310929226747601816782662354e-02 -1.55961726942718093003037438393221e-01
            1.95970988315751104558160022861557e-01 9.56038125793923332373935863870429e-01
            -2.84201318985743855982661898451624e-01 9.33618303440851171615122439106926e-01
            4.40270432259555588583310736794374e-01 -6.78727075977861415978509285196196e-01
            -4.55171387624477702393832601046597e-02 9.74854715489385070803507460368564e-01
            3.59370384417379631902811354393634e-02 -7.69674727765501720533336538210278e-01
            3.03403495435356662657255810700008e-01 -2.98721147700096223420729302233667e-01
            -3.94779814509269799671642431349028e-01 -4.68832544170729725863111525541171e-01
            2.05799163272229468368834659486311e-01 5.50579882677324716766520396049600e-01
            -1.52763465750910282930918882016158e-02 -9.82168456110988907781234047433827e-01
            ];
        
    case   18
        
        % DEGREE.             :          18
        % CARDIN.             :         190
        % LEBESGUE CONST.     : 7.47591e+01
        % ABS.DET.VAND.MATRIX : 1.11798e+176
        % CONDITIONING VAND.  : 1.36738e+03
        % POINTS OUT DISK     :           0
        
        pts=[
            9.99516282291988078512190440960694e-01 3.10998622698369156513997779711644e-02
            -1.00000000000000000000000000000000e+00 -0.00000000000000000000000000000000e+00
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            -5.77379444863542756749552609107923e-01 2.05640543343572512746675329253776e-01
            6.58418280570842839871659180062125e-01 2.57847567004490074715761238621781e-01
            -8.39358674760744216669650086259935e-01 2.98947556019653504222333140205592e-01
            8.69602986211723494314185245457338e-01 2.50005282436187581396325185778551e-01
            3.05438557953331024652499081639689e-01 4.40247250454620375759162698159344e-01
            -3.21613039335863215573141360437148e-01 -3.58680436377076972132016408068012e-01
            -9.43618424191435267722738444717834e-01 -1.78180336394002186706231327661953e-01
            9.52031820332836509201968056004262e-01 -2.41916630304473939672149640500720e-01
            4.79169353634575578837484499672428e-01 -8.50182484394046311670933846471598e-01
            -7.12583964147506909725393597909715e-01 7.01586839984776666767629649257287e-01
            -1.56040645903459479493236017333402e-01 -5.12602839480523320148108723515179e-01
            7.74474761895350694018702597531956e-01 -6.31824506806499042177449609880568e-01
            -9.80053383657033649889456228265772e-01 1.22570891038516954774095779612253e-01
            1.61407992027690960767927208507899e-01 -3.93999321363938448392616464843741e-01
            9.82635248222263668615994447463891e-01 -1.85547753829493428101571339539078e-01
            -4.61407394332907061507142998380004e-01 7.62230832885388953634731024067150e-01
            -2.31176322114969662324668320252385e-01 9.72911870671437473134801621199585e-01
            7.74857107102889908212262071174337e-01 6.32136427974326342571487202803837e-01
            -7.74857107102889908212262071174337e-01 -6.32136427974326342571487202803837e-01
            5.17849624898325311406210857967380e-01 8.55471662881163830149944260483608e-01
            -9.31144062097659652721404199837707e-01 3.64651526282654958599493966175942e-01
            9.69198999199666122450480543193407e-01 2.46278906832001348581329125408956e-01
            1.39515338943923122849710694026726e-01 9.90219910019669469214420587377390e-01
            -4.90994080973322200023289951786865e-01 -8.71162908099950450413473390653962e-01
            8.63421704896663477768470329465345e-01 -5.04482863448639795578287703392562e-01
            -8.78693400099268773217886518978048e-01 -4.77386540051127017392218476743437e-01
            3.20825769815367700665831307560438e-01 -9.47138229311000112708995857246919e-01
            -6.44110411450397823962532584118890e-01 -7.64932531574647578409553716483060e-01
            6.20009547460774967930774437263608e-01 -7.84594265246366107113829002628336e-01
            -9.87930439740756671795907095656730e-01 -1.54898180214084663086637760898157e-01
            9.31144062097659541699101737322053e-01 3.64651526282655180644098891207250e-01
            -7.76838472890061110254222853654937e-02 9.96978043825629267615795470192097e-01
            -3.50126449191390465287554434326012e-01 9.36702444523675126220041420310736e-01
            7.06965026751690372996961286844453e-01 -6.96054618197665009304841987614054e-01
            1.55518119203506659242730592040971e-02 -9.99879063260149303360435624199454e-01
            -5.95308864276666072079535751981894e-01 8.03496954638675653370682994136587e-01
            6.20009547460775078953076899779262e-01 7.84594265246365774046921615081374e-01
            -8.63421704896663477768470329465345e-01 5.04482863448639795578287703392562e-01
            3.50126449191390798354461821872974e-01 9.36702444523675015197738957795082e-01
            8.63421704896663477768470329465345e-01 5.04482863448639795578287703392562e-01
            -3.20825769815367811688133770076092e-01 -9.47138229311000112708995857246919e-01
            -9.52013107532729940807314505946124e-01 -3.06057254587887295205206328319036e-01
            9.50596473416901455166794221440796e-01 2.95777266676101845133128875886541e-02
            5.09041415750371206705437998607522e-01 0.00000000000000000000000000000000e+00
            -7.06764741115032180118760152254254e-01 -2.19909235049693362051836942328009e-02
            -1.39515338943923122849710694026726e-01 -9.90219910019669469214420587377390e-01
            7.70513242775789253258267308410723e-01 0.00000000000000000000000000000000e+00
            -9.76389420563607002279127300425898e-01 2.16017821976483392898416013849783e-01
            1.70243555722398520746452277307981e-01 -9.85402015288681321081298847275320e-01
            9.82287250728688721146397710981546e-01 0.00000000000000000000000000000000e+00
            -4.53112299215529645834976690821350e-01 -2.82244247661442744512516611621322e-02
            -2.10498838597022175367357021968928e-01 4.02240739859935525490897134659463e-01
            3.51452384734242395669667757829302e-01 -3.68245005687458604182182853037375e-01
            -6.05326788023288986373415809794096e-01 -4.06170825473795626958661841854337e-01
            6.29407335104375631473772045865189e-01 -3.67752180525778238351364279878908e-01
            -8.42192885067440166757535280339653e-01 -2.42125054117246912843697259631881e-01
            8.63562631523739021588426112430170e-01 -2.19436112757292800523600817541592e-01
            -3.48497060737323338663884442212293e-01 4.41007394853088041664079810288968e-01
            1.40293757567952454001769524438714e-01 4.60873371001152720705107412868529e-01
            -9.18554492564195745885058386193123e-01 1.44020685669936754358388952823589e-01
            7.65528996461353306024477660685079e-01 -3.56167097525037057437202747678384e-01
            -2.74145336308877600461908485840468e-02 5.87145591960228641070784760813694e-01
            9.42034263469989086026146196672926e-01 -3.35516685797525027190602031623712e-01
            -7.26750241965370524788170314423041e-01 3.94837133696432251994679063500371e-01
            4.88200678413237776354094421549235e-01 5.11527220777989954925146776076872e-01
            -9.78013660706298804115021994221024e-01 -9.15288064818417651524740108470723e-02
            -4.72056100986557924947817355132429e-01 -5.26462759861861973043062334909337e-01
            2.97297037241842330990149179115178e-02 -6.36730309804402150319901920738630e-01
            5.17891700125356790884723068302264e-01 -5.42636898601099204597630887292325e-01
            -4.90994080973322089000987489271211e-01 8.71162908099950561435775853169616e-01
            -7.43183348431955104196333650179440e-01 -4.34229602473535980511343268517521e-01
            7.70004896217444501083093655324774e-01 4.18337014009830032712500269553857e-01
            2.40654961361221053639525280232192e-01 -7.10458870276373666285962826805189e-01
            -8.60325670843451795199996467999881e-01 2.67689584896827550974940379546752e-02
            9.52874801707685237239786602003733e-01 2.10815413344350582880437627864012e-01
            -2.48689887164854850354345217056107e-01 -0.00000000000000000000000000000000e+00
            -6.07125549517514295772002697049174e-01 5.61659009956570809407594424556009e-01
            6.49707826675848920672251551877707e-01 5.64585314197068699471060426731128e-01
            -9.29326736537534059401366448582849e-01 -2.89159206528575851713469546666602e-02
            3.66522970851886398246932685651700e-01 3.43015761620598172432217154437240e-02
            8.60742027003943643848060673917644e-01 0.00000000000000000000000000000000e+00
            -1.25090788598041458001830505963881e-01 -7.79192168880785476781092313558474e-03
            1.87025573288107538338920221576700e-01 7.87102237402313886782678764575394e-01
            -2.83258247575991439770604074510629e-01 7.57808196292209634847836241533514e-01
            6.54050938260793635059542339149630e-01 -6.05070240803835979193081584526226e-01
            -5.94825388403371735357438865321456e-01 -6.63381778056206794502713819383644e-01
            3.83487100304596206790108681161655e-01 7.32802784002839047161614871583879e-01
            -8.62296885495731801896113211114425e-01 -4.01188956044199296346874916707748e-01
            8.35687083989954060392335577489575e-01 -4.54021579707221623056767612069962e-01
            -1.37729929762238745283298158028629e-01 7.97206976659864952061695930751739e-01
            3.85753608660151098286306137197244e-02 8.26180499574031168563692517636809e-01
            -7.55590219348507896590660948277218e-01 5.41818912678008635097626211063471e-01
            7.55590219348507785568358485761564e-01 -5.41818912678008524075323748547817e-01
            5.21673134454968212558867435291177e-01 7.04109748744772723227924871025607e-01
            -9.29085549226586260296301134076202e-01 2.98686404865274202080627219402231e-01
            9.33607640519270698398202057433082e-01 -1.16762129785023940908850192954560e-01
            -3.13536654749127896568694495726959e-01 -7.65347661178281124527700285398168e-01
            3.91484068092108616898627815317013e-01 -7.48084133111102222990496102283942e-01
            -9.17380444830022401259128628225881e-02 -8.39329361441008225952487009635661e-01
            6.60032621755781589989453550515464e-01 -4.11135188963075021151283294784662e-02
            -7.29048150760613689058686759381089e-01 -5.94765008436461517504767471109517e-01
            1.84127484570157706755821891420055e-01 3.47681820309988925954414185071073e-02
            -6.05509535035596679897196281672223e-01 -9.49381872512224250781898149398330e-02
            8.43803723761331614383607302443124e-01 4.58431280038260335896183050863328e-01
            4.01453441623211698674467129421828e-02 -8.59805320053875110630769995623268e-01
            -8.43803723761331614383607302443124e-01 4.58431280038260335896183050863328e-01
            -4.61407394332906894973689304606523e-01 -7.62230832885388953634731024067150e-01
            7.72883604465182894038832728256239e-01 5.54219659644423323641149181639776e-01
            5.30424081977676986632275202282472e-01 -7.15921028748545018594029443192994e-01
            -1.44126732539939655941552132389916e-02 -3.08680705037097657950084794720169e-01
            4.78554956670983411282094266425702e-01 -2.41030097006778715185149053468194e-01
            -4.88200678413237776354094421549235e-01 5.11527220777989954925146776076872e-01
            2.09174590139223043339100627235894e-01 -8.80316980248837754885471440502442e-01
            -4.30261440443399589828032958394033e-01 2.16706472856886461775260954709665e-01
            -7.40517886824198390272044889570680e-01 -2.12894144086411690741655888814421e-01
            -5.98878714882386531392910455906531e-01 7.11216281149079865286921631195582e-01
            -1.76149655129986926205276631662855e-01 -2.53895257557349374177846357270028e-01
            7.89915634363865803990734093531501e-01 1.74762089066837156270395325918798e-01
            -9.14650940790253441647905674471986e-01 -3.58192545226209524411586926362361e-01
            3.72533398339925547748663348102127e-01 -5.74302714042559392382258920406457e-01
            -3.06626195219769470678983225297998e-01 -5.85929824827935763487118947523413e-01
            3.35753608517890198004174351353868e-02 3.07187561525850971744233675053692e-01
            2.98296256841325713171642064480693e-01 8.80626854499202527115642169519560e-01
            -4.76261511466449072305096024138038e-01 -2.98517497566481826432038815255510e-01
            6.56628459545393616814124015945708e-01 6.88002589604513770105143066757591e-01
            -2.42970979450412988276397641129734e-01 8.97468337522618564428000809130026e-01
            -7.40517886824198501294347352086334e-01 2.12894144086411885030685198216815e-01
            4.85314988399162206000880814826814e-01 2.83561431955726639397852295587654e-01
            1.29718081570789151024669649814314e-01 9.20683188194668877812887330946978e-01
            8.75881307075453530153197334584547e-01 -3.11955525077695416946710338379489e-01
            -8.24392853946584791557938842743170e-01 1.82389879500867824191345789586194e-01
            -8.26908623205479309525856024265522e-01 -5.18300738086902801349253877560841e-01
            -4.38830463720388391779536618741986e-02 9.39856850604922744452096594613977e-01
            2.48653459370298007824473529581155e-01 2.30032249381013992328703920975386e-01
            -2.45872772057113470056677329012018e-01 -9.08186765675896290339608185604447e-01
            7.20909951093238854369360524287913e-01 -2.07256988294474808753875549882650e-01
            -5.95391253486001303940611251164228e-01 -7.53440918734425757463668560376391e-01
            -4.71498515663434869527037562875194e-01 8.36572239844345588721807871479541e-01
            4.90994080973322089000987489271211e-01 -8.71162908099950561435775853169616e-01
            -7.56196038929077385404298183857463e-01 6.16912535892230939893465802015271e-01
            3.56676988906966196957171177928103e-01 -8.70653861745386503656618515378796e-01
            7.93086479827163404188183903897880e-01 -5.68706744947136910717233604373178e-01
            1.47906520670461495009506691644674e-02 -9.50941498620659242391184307052754e-01
            4.97287724920035256737094186973991e-01 8.21504036140336180160659296234371e-01
            -4.18577054746865184764459399957559e-01 -8.64266863873912050131309570133453e-01
            -2.85243925946794985026855329124373e-01 2.32704939783739156444752893548866e-01
            6.26163684451213065074171026935801e-01 4.49009685382558310084988306698506e-01
            -1.21607876015710131278879657656944e-01 2.51092735338839023739865297102369e-01
            2.05799163272229190813078503197175e-01 -5.50579882677324605744217933533946e-01
            -7.16380022460453069754748867126182e-01 -6.62731612081821741533360636822181e-01
            8.15679528628174721482935183303198e-01 5.47316315781874451751320975745330e-01
            -1.20380372704617932821946624244447e-01 -6.96784447205516466183894408459309e-01
            2.31542775213905482889487075226498e-01 6.19452440841176121821831657143775e-01
            -5.87172973628806160739657116209855e-01 3.93989719459664822220901214677724e-01
            4.93912508694651808394127101564663e-01 -4.02938923984880403850183938629925e-01
            2.24691794306768090339065935268081e-02 -4.81229403181283044155236439110013e-01
            -6.05077709929027651725164105300792e-01 7.65698694774944343954814485186944e-01
            6.26355280389529367468526288575958e-01 1.18272589534903327379566917443299e-01
            -2.98383417186724730907343428043532e-01 6.16094210801656827491967760579428e-01
            -9.55048215067225841901077387774421e-02 6.77852173636931287781237642775523e-01
            1.63483811587410099441797228791984e-01 -9.46275333135055074507135941530578e-01
            -3.80269878878140565880983103852486e-01 -1.91527607262398313903162261340185e-01
            4.89903651646915017092709376811399e-01 6.19951091365638085051159578142688e-01
            5.31782528390954273289636944355152e-02 6.82478434575279901963540396536700e-01
            -6.24299389990488395874024263321189e-01 -5.42505804577096628626975416409550e-01
            5.83495548776899797260853119951207e-01 -1.87584650158581195400486762991932e-01
            1.50610461212588253587796316423919e-01 -1.57806725903307409897280422228505e-01
            -4.09181476770282881449958267694456e-01 -6.75955222370918429319885945005808e-01
            3.46785941853529622314056268805871e-01 -1.23511929877459472604250834137929e-01
            -4.70386283019887274647885533340741e-01 6.34887146136352331637908719130792e-01
            6.49602919856937188747281197720440e-01 -6.80641365111786522668069210340036e-01
            -4.08797552610881478885040962722996e-02 -1.50998633974671225832508980602142e-01
            2.25608847715350141482915091728501e-01 9.49481002477438607023429995024344e-01
            -2.55028445125500879964164369084756e-01 9.42005317612315429265379407297587e-01
            -5.83241597269268718939372320164694e-01 7.29434166139660555838730715549900e-02
            4.81484404456514136771971834605210e-01 -7.95397436490627329774838472076226e-01
            6.32701445229335801556658225308638e-01 7.51383473433396398277750449778978e-01
            3.59370384417379631902811354393634e-02 -7.69674727765501720533336538210278e-01
            -1.51772739315890383554608078497949e-02 9.75798737747193167813009040401084e-01
            -1.66143539641541965190540963703825e-01 -9.61670343968445862081750874494901e-01
            -4.36747737946006742149052115564700e-01 -4.30007522976771805467421927460236e-01
            4.00462417288204863830003432667581e-01 8.96949327043514488266851003572810e-01
            4.53828467556370462876458304890548e-01 1.61636396105388380206591136811767e-01
            6.01167545009807513523014677048195e-02 1.77476006502690797361765362438746e-01
            1.91201445253530166734634576641838e-01 -8.04676508713470628464392575551756e-01
            -7.44312615320496973048136624129256e-01 9.30878480625379894908277833565080e-02
            -6.72120271297810129018657221422473e-02 4.77042079820645559262715096338070e-01
            ];
        
    case   19
        
        % DEGREE.             :          19
        % CARDIN.             :         210
        % LEBESGUE CONST.     : 1.48390e+02
        % ABS.DET.VAND.MATRIX : 2.54107e+198
        % CONDITIONING VAND.  : 2.18408e+03
        % POINTS OUT DISK     :           0
        
        pts=[
            9.99516282291988078512190440960694e-01 3.10998622698369156513997779711644e-02
            -1.00000000000000000000000000000000e+00 -0.00000000000000000000000000000000e+00
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            -5.77379444863542756749552609107923e-01 2.05640543343572512746675329253776e-01
            6.58418280570842839871659180062125e-01 2.57847567004490074715761238621781e-01
            -8.39358674760744216669650086259935e-01 2.98947556019653504222333140205592e-01
            8.69602986211723494314185245457338e-01 2.50005282436187581396325185778551e-01
            3.05438557953331024652499081639689e-01 4.40247250454620375759162698159344e-01
            -3.21613039335863215573141360437148e-01 -3.58680436377076972132016408068012e-01
            -9.43618424191435267722738444717834e-01 -1.78180336394002186706231327661953e-01
            9.52031820332836509201968056004262e-01 -2.41916630304473939672149640500720e-01
            4.79169353634575578837484499672428e-01 -8.50182484394046311670933846471598e-01
            -7.12583964147506909725393597909715e-01 7.01586839984776666767629649257287e-01
            -1.56040645903459479493236017333402e-01 -5.12602839480523320148108723515179e-01
            7.74474761895350694018702597531956e-01 -6.31824506806499042177449609880568e-01
            -9.80053383657033649889456228265772e-01 1.22570891038516954774095779612253e-01
            1.61407992027690960767927208507899e-01 -3.93999321363938448392616464843741e-01
            9.82635248222263668615994447463891e-01 -1.85547753829493428101571339539078e-01
            -4.61407394332907061507142998380004e-01 7.62230832885388953634731024067150e-01
            -8.92674317839844744248978258838179e-01 -4.49606412858295056977908643602859e-01
            -2.31176322114969662324668320252385e-01 9.72911870671437473134801621199585e-01
            7.74857107102889908212262071174337e-01 6.32136427974326342571487202803837e-01
            -7.74857107102889908212262071174337e-01 -6.32136427974326342571487202803837e-01
            5.39912970229085575901706306467531e-01 8.32337410635087482368987821246265e-01
            -9.19353038982236192566688259830698e-01 3.93433589966752705890939978417009e-01
            9.69198999199666122450480543193407e-01 2.46278906832001348581329125408956e-01
            -4.63663531985327337370250688763917e-01 -8.86011359468314774190389471186791e-01
            1.39515338943923122849710694026726e-01 9.90219910019669469214420587377390e-01
            -6.67588140216153647088503930717707e-01 -7.44530775081015172567333593178773e-01
            8.63421704896663477768470329465345e-01 -5.04482863448639795578287703392562e-01
            2.91214712227251926179860674892552e-01 -9.56657719031419717659048274072120e-01
            -9.87930439740756671795907095656730e-01 -1.54898180214084663086637760898157e-01
            6.20009547460774967930774437263608e-01 -7.84594265246366107113829002628336e-01
            -7.76838472890061110254222853654937e-02 9.96978043825629267615795470192097e-01
            4.30517957975215970201077197998529e-01 8.88922125952789521363683888921514e-01
            9.31144062097659541699101737322053e-01 3.64651526282655180644098891207250e-01
            -3.50126449191390465287554434326012e-01 9.36702444523675126220041420310736e-01
            -5.70032258213782783151657440612325e-01 -8.21622312620400530036590680538211e-01
            7.06965026751690372996961286844453e-01 -6.96054618197665009304841987614054e-01
            4.66403903874177058752614755121613e-02 -9.98911744842610804795413059764542e-01
            -6.20009547460774967930774437263608e-01 7.84594265246366107113829002628336e-01
            6.67588140216153647088503930717707e-01 7.44530775081015172567333593178773e-01
            -2.61321923212860740548535432026256e-01 -9.65251704193435644540954854164738e-01
            -8.30387982764797305534898441692349e-01 5.57185604695428882138230619602837e-01
            8.63421704896663477768470329465345e-01 5.04482863448639795578287703392562e-01
            2.61321923212860740548535432026256e-01 9.65251704193435644540954854164738e-01
            -9.52013107532729940807314505946124e-01 -3.06057254587887295205206328319036e-01
            9.59829174616289138022295901464531e-01 2.98650013631469937736007125295146e-02
            -1.08652150085474491492121273950033e-01 -9.94079830940052588950095469044754e-01
            5.09041415750371206705437998607522e-01 0.00000000000000000000000000000000e+00
            -7.28968627421411330402634121128358e-01 -0.00000000000000000000000000000000e+00
            7.70513242775789253258267308410723e-01 0.00000000000000000000000000000000e+00
            -4.53112299215529645834976690821350e-01 -2.82244247661442744512516611621322e-02
            -9.69198999199666011428178080677753e-01 2.46278906832001820426114591100486e-01
            9.17754625683981140049638725031400e-01 0.00000000000000000000000000000000e+00
            1.70243555722398520746452277307981e-01 -9.85402015288681321081298847275320e-01
            -8.91006524188367787786546614370309e-01 -0.00000000000000000000000000000000e+00
            3.73290258250643602888629857261549e-01 -3.04534046710166150795373596338322e-01
            -2.91599183060326117011840096893138e-01 4.49533392147918209325041516422061e-01
            -8.11459033367125659452767649781890e-01 -2.33289268826120610444974090569303e-01
            6.29407335104375631473772045865189e-01 -3.67752180525778238351364279878908e-01
            -1.40293757567952287468315830665233e-01 4.60873371001152776216258644126356e-01
            8.49315557290494571596184414374875e-01 -2.15815851210783504754431305627804e-01
            -5.92402124006527253108345121290768e-01 -4.24799933188803047201531626342330e-01
            1.32208712735558592266116306745971e-01 4.34313515942769179378757371523534e-01
            -9.39577682361783184106229782628361e-01 1.47316923654866260839924052561400e-01
            9.42034263469989086026146196672926e-01 -3.35516685797525027190602031623712e-01
            5.03872953213493790514121428714134e-01 4.96096812144476817518778943849611e-01
            -7.10877893525413528941214735823451e-01 -3.86213823787218191174730463899323e-01
            7.65528996461353306024477660685079e-01 -3.56167097525037057437202747678384e-01
            -4.40923918056906782858561655302765e-01 -5.23632350720130257393236661300762e-01
            -9.78013660706298804115021994221024e-01 -9.15288064818417651524740108470723e-02
            9.14112569320840728792632035037968e-03 -5.87714167460328562597737800388131e-01
            9.11900699636647549262136180914240e-01 -2.31719087237299919967981054469419e-01
            -1.02845977502097063460695380854304e-02 6.61231888422635250002201701136073e-01
            -4.90994080973322089000987489271211e-01 8.71162908099950561435775853169616e-01
            5.00765253930148146466194702952635e-01 -5.58480776068815298884828735026531e-01
            -6.86797769655803524990744790557073e-01 4.60837389909014294442357595471549e-01
            7.43183348431954993174031187663786e-01 4.34229602473535925000192037259694e-01
            2.47200504266477855130901275515498e-01 -7.29782548423337673604294195683906e-01
            -8.65755853912873574174113855406176e-01 -3.39044414680874484080419506426551e-01
            9.22910355552496897146852461446542e-01 2.65330809302220527179372311366023e-01
            -2.18037721647796084445403153040388e-01 6.78422476252822908432404958034567e-03
            -5.89364354285963787383195722213713e-01 5.80268846518084191465902677009581e-01
            6.19786114784536157351624297007220e-01 5.73371448280105777683957057888620e-01
            -9.51056516295153531181938433292089e-01 -0.00000000000000000000000000000000e+00
            3.78317441514472774244381980679464e-01 -6.71242557306079734580350759642897e-01
            9.70431475564132339783895986329298e-01 1.52154507585370191158347097371006e-01
            -3.22133251809616327765439791619428e-01 -7.21508913084924929925989545154152e-01
            -7.92409263584862078033665966358967e-01 -4.62991481508289282054136037913850e-01
            1.82665129661720521125900518200069e-01 7.68751191210845674639529079286149e-01
            -2.89582784665928405498647180138505e-01 7.74728395741027142662460391875356e-01
            6.34916961087438669331106666504638e-01 -6.25118451711136446924399479030399e-01
            -6.34916961087438669331106666504638e-01 -6.25118451711136446924399479030399e-01
            3.60511535149943440625719404124538e-01 7.44374710321468513463116778439144e-01
            -8.20687135869564876600179559318349e-01 1.02639667578131574265931647005345e-01
            8.26745721960402857320104885729961e-01 -4.49163814891701540510382528736955e-01
            -1.37729929762238745283298158028629e-01 7.97206976659864952061695930751739e-01
            -4.55449580189088043535861061172909e-01 3.71560469800152337604970398388105e-01
            7.38374233419929359989453132584458e-01 -5.65055577029585087700525036780164e-01
            -9.01890459132705490929993175086565e-01 3.53195328037234024254331643533078e-01
            4.99523075713226405447642264334718e-01 7.19993121022344584325480809638975e-01
            -1.31308290965072901185228815279515e-02 -8.44225815235339904774036767776124e-01
            9.71670887696349683260166330001084e-01 -9.09352090028796716802972355253587e-02
            -7.20443918107661396277308085700497e-01 -5.87745586603920844304127513169078e-01
            -1.43741388233179501199288097268436e-01 -8.32002439354197043108740672323620e-01
            2.48208820734276941477602917984768e-01 1.54609601179148212907188408848924e-02
            8.35687083989954060392335577489575e-01 4.54021579707221623056767612069962e-01
            -7.01816619833756538682223435898777e-01 6.09866990282164844749956955638481e-01
            4.05347466485949103898178691451903e-01 1.30312841036781729142646213404078e-01
            -4.61407394332906894973689304606523e-01 -7.62230832885388953634731024067150e-01
            4.88089809950044140052227703563403e-01 -3.27505481229636796758342143220943e-01
            -8.05844171502799855311138799152104e-01 5.05097682079852616787718488922110e-01
            6.60032621755781589989453550515464e-01 -4.11135188963075021151283294784662e-02
            -3.38082664641408681127643376385095e-01 2.10592136859435383189342871901317e-02
            7.47192607105389705601794503309065e-01 5.71804012993040622347962198546156e-01
            -5.83241597269268496894767395133385e-01 -7.29434166139658335392681465236819e-02
            1.33860981159068352652763422838689e-02 8.60637931669345546126237422868144e-01
            -9.25348246756160897064091841457412e-01 -3.29573762865652042375330665890942e-01
            -4.80577417671124336451260461444690e-03 3.08979622867089354087966057704762e-01
            3.43009443070938557784899103353382e-01 -8.37291178048896633789865973085398e-01
            -5.83355059773855266769260197179392e-01 7.38209655602076519897991602192633e-01
            6.28120842711295979121643995313207e-01 7.00514688168311083771300218359102e-01
            -3.21330168319706233059918076833128e-01 -8.59663001351095301671989545866381e-01
            -7.20909951093238965391662986803567e-01 2.07256988294474836509451165511564e-01
            4.92410666622814907622540658849175e-01 -7.59107192872271241412818199023604e-01
            -1.68168340881542005549320606405672e-01 -2.59250673938710485266767591383541e-01
            7.89915634363865803990734093531501e-01 1.74762089066837156270395325918798e-01
            -8.42627714430384422428232937818393e-01 -4.92333282550383777298463883198565e-01
            1.81695671846368811408467536239186e-01 -8.86396456280507338831853303418029e-01
            8.84118930031238425826245475036558e-01 1.10572798222052334771170478688873e-01
            -4.65076024826899137032398812152678e-01 5.88532843529876226185137966240291e-01
            2.12163138969623549234810866437329e-01 8.92894369691566880398170269472757e-01
            -2.70764591773612728342612854248728e-01 8.89477852198903629066251141921384e-01
            5.16483021865079883205851274396991e-01 2.80600767884982560396167627914110e-01
            -4.85314988399162206000880814826814e-01 -2.83561431955726639397852295587654e-01
            5.26798842594551706280947556138017e-03 -3.38696954385553194999403103793156e-01
            -6.93987688285326642656514195550699e-01 -2.23106136789297637701423582257121e-01
            7.20909951093238854369360524287913e-01 -2.07256988294474808753875549882650e-01
            3.25539339545702743894395325696678e-01 8.70923907192157398249321431649150e-01
            -8.54088381272911090213995066733332e-01 -1.06817011872997802424123392484034e-01
            6.18535160994594601646667797467671e-01 -7.34559880040012935786819525674218e-01
            -1.29718081570789040002367187298660e-01 9.20683188194668877812887330946978e-01
            1.15225489162813748977853833821428e-01 4.93103040233457759478064019731391e-02
            -1.02228718520951153414522138973553e-01 -9.35310595736763117358236740983557e-01
            3.00933735530409862679590560219367e-01 -5.33941891255299561080960302206222e-01
            -5.71671343392394670779310672514839e-01 -7.71593052000173251769865601090714e-01
            6.22883217959981916322931283502840e-01 4.17951089920782337117088900413364e-01
            -3.22745376284550800072281617758563e-01 2.31434373252115077601587245226256e-01
            1.71700477012344449034841886714275e-01 -1.79904436220460151041322660603328e-01
            -5.74635921652868697506733042246196e-01 4.12060138264013775533101124892710e-01
            8.10389551292669318982575532572810e-01 -5.43766771133245896407970576547086e-01
            1.46324007582510282055032746484358e-02 -9.40766981901440146884851856157184e-01
            -8.48128132722420713385247381665977e-01 4.95547084976700857872344840870937e-01
            4.92504260220553269356003056600457e-01 8.13601899488981650243601961847162e-01
            1.47906520670462657274235596105427e-02 9.50941498620659242391184307052754e-01
            -4.45253162044179140455213428140269e-01 -8.50831113935466731845735921524465e-01
            4.90994080973322089000987489271211e-01 -8.71162908099950561435775853169616e-01
            3.90251516094856143457292318998952e-01 -7.36898990454048635756478802250058e-02
            -2.95550258459056580662149826821405e-01 -5.64764895714953696348459288856247e-01
            8.48128132722420713385247381665977e-01 4.95547084976700857872344840870937e-01
            -6.95422634900311420480534252419602e-01 6.84690357096781387902240112453001e-01
            -1.60024334624274483518391321013041e-01 2.64355282036475569551470243823132e-01
            6.22978435974130828611805554828607e-01 -2.21881165331970831022800894061220e-01
            -5.86017682095639758088623239018489e-01 -2.50784008617757347980870008541388e-01
            2.23178998160118241544935813180928e-01 5.97076609395631674459536952781491e-01
            -2.31542775213905260844882150195190e-01 6.19452440841176232844134119659429e-01
            -4.45253162044179251477515890655923e-01 8.50831113935466842868038384040119e-01
            3.66174981400515853202648486330872e-01 6.04909813936203133444280410913052e-01
            6.56819132636027092075892142020166e-01 -5.02643913069278025496089412627043e-01
            -7.36644358615534677880987146636471e-01 -6.40131717034015723655215879261959e-01
            7.49214253156784467518880532566072e-03 4.81695412382958321195758344401838e-01
            5.10114132187257762751642076182179e-01 -1.63993677805898740995615980864386e-01
            -1.84782503976524070665732324414421e-01 -6.82536025587049732799016510398360e-01
            8.90612936721255410432718235824723e-01 -4.14363173905291481968760081144865e-01
            2.82065666536236270278692472857074e-01 -9.26602557617558164615445548406569e-01
            -5.01598260570487619602886297798250e-01 -6.34750094273435205849409612710588e-01
            2.13351519613370538275276544482040e-01 2.23545592421751565526122362825845e-01
            6.78190783855465717522292834473774e-01 7.10595175636335674163035491801566e-01
            -3.17846728893326102571847968647489e-01 -1.85712528444115898551558530016337e-01
            -7.16411900403769119627384043269558e-01 3.33315064979498643094046883561532e-01
            4.89903651646915017092709376811399e-01 6.19951091365638085051159578142688e-01
            1.13367829894939712437329504268746e-02 -7.28880468332157693822637156699784e-01
            -2.82065666536236547834448629146209e-01 -9.26602557617558053593143085890915e-01
            1.21524310929226556782234425213574e-02 -1.55961726942718093003037438393221e-01
            1.60166050011323779500571617973037e-01 -5.91609578050713125207948905881494e-01
            -5.29359601188480355382637299044291e-01 8.29985954521920143989177631738130e-02
            6.76284923383391745055348565074382e-01 1.06035100979172125512661750690313e-01
            -6.32496620949222765339925445005065e-01 7.91035220135796052209897766260838e-02
            1.94498349403579323668722622642235e-01 9.48853904508599743650165692088194e-01
            -2.25608847715350030460612629212847e-01 9.49481002477438718045732457539998e-01
            4.07683614941688809985009811498458e-01 9.13123250228618843493677559308708e-01
            3.59370384417378938013420963670796e-02 7.69674727765501831555639000725932e-01
            -5.64438256239784941215020808158442e-01 6.70315487201727400190520711475983e-01
            -5.84764307624744406055583567649592e-01 7.89264814540898584915851188270608e-01
            -7.58129687012310954363769610608870e-02 -9.72967584254334605375902356172446e-01
            -1.45193743345031101021902486536419e-01 -1.18450554871353488928242825295456e-01
            3.91484068092108616898627815317013e-01 -7.48084133111102222990496102283942e-01
            -6.10664585519031799520917047630064e-01 -5.30657385871847409042345589114120e-01
            1.66083629135901272855235788483697e-01 -8.10233611050163338163088155852165e-01
            -4.08797552610881478885040962722996e-02 1.50998633974671336854811443117796e-01
            1.36155357822940165091196718094579e-01 -9.66372208193673531972933687939076e-01
            5.40202000534838910894563923648093e-01 1.55304611250983354286958615375625e-01
            -5.37504126211233668186650902498513e-01 -8.28623902897734909700488969974685e-01
            -2.40654961361221081395100895861106e-01 7.10458870276373777308265289320843e-01
            2.24930761886806662719351379564614e-01 8.30832708436468836410426774818916e-01
            3.21387976289739751845075943492702e-01 2.79280815349069122532910114387050e-01
            3.74244788498352826122683723042428e-02 -4.80297835611737944372379160995479e-01
            2.77477704790345447438681958374218e-01 -4.58384639316966879807324630746734e-01
            -1.18031435980611457137179343135358e-01 5.75812541486273365975989690923598e-01
            ];
        
    case   20
        
        % DEGREE.             :          20
        % CARDIN.             :         231
        % LEBESGUE CONST.     : 1.06119e+02
        % ABS.DET.VAND.MATRIX : 4.68313e+222
        % CONDITIONING VAND.  : 1.49841e+03
        % POINTS OUT DISK     :           0
        
        pts=[
            9.99516282291988078512190440960694e-01 3.10998622698369156513997779711644e-02
            -1.00000000000000000000000000000000e+00 -0.00000000000000000000000000000000e+00
            0.00000000000000000000000000000000e+00 0.00000000000000000000000000000000e+00
            -5.77379444863542756749552609107923e-01 2.05640543343572512746675329253776e-01
            6.58418280570842839871659180062125e-01 2.57847567004490074715761238621781e-01
            -8.39358674760744216669650086259935e-01 2.98947556019653504222333140205592e-01
            8.69602986211723494314185245457338e-01 2.50005282436187581396325185778551e-01
            3.05438557953331024652499081639689e-01 4.40247250454620375759162698159344e-01
            -3.21613039335863215573141360437148e-01 -3.58680436377076972132016408068012e-01
            -9.43618424191435267722738444717834e-01 -1.78180336394002186706231327661953e-01
            9.52031820332836509201968056004262e-01 -2.41916630304473939672149640500720e-01
            4.79169353634575578837484499672428e-01 -8.50182484394046311670933846471598e-01
            -7.12583964147506909725393597909715e-01 7.01586839984776666767629649257287e-01
            -1.56040645903459479493236017333402e-01 -5.12602839480523320148108723515179e-01
            7.74474761895350694018702597531956e-01 -6.31824506806499042177449609880568e-01
            -9.80053383657033649889456228265772e-01 1.22570891038516954774095779612253e-01
            1.61407992027690960767927208507899e-01 -3.93999321363938448392616464843741e-01
            9.82635248222263668615994447463891e-01 -1.85547753829493428101571339539078e-01
            -4.61407394332907061507142998380004e-01 7.62230832885388953634731024067150e-01
            -8.92674317839844744248978258838179e-01 -4.49606412858295056977908643602859e-01
            5.71671343392394670779310672514839e-01 -7.71593052000173251769865601090714e-01
            5.17849624898325311406210857967380e-01 8.55471662881163830149944260483608e-01
            -7.74474761895350805041005060047610e-01 -6.31824506806498931155147147364914e-01
            -2.31176322114969662324668320252385e-01 9.72911870671437473134801621199585e-01
            8.93115016186791454977367266110377e-01 -4.49828376007636032696268557629082e-01
            -9.19353038982236192566688259830698e-01 3.93433589966752705890939978417009e-01
            1.39515338943923122849710694026726e-01 9.90219910019669469214420587377390e-01
            9.76389420563607002279127300425898e-01 2.16017821976483503920718476365437e-01
            -4.63663531985327337370250688763917e-01 -8.86011359468314774190389471186791e-01
            6.90420020571746673176960484852316e-01 -7.23408733147249849437798729923088e-01
            -6.67588140216153647088503930717707e-01 -7.44530775081015172567333593178773e-01
            -7.76838472890061110254222853654937e-02 9.96978043825629267615795470192097e-01
            -9.87930439740756671795907095656730e-01 -1.54898180214084663086637760898157e-01
            7.74857107102889908212262071174337e-01 6.32136427974326342571487202803837e-01
            3.20825769815367700665831307560438e-01 -9.47138229311000112708995857246919e-01
            9.31144062097659541699101737322053e-01 3.64651526282655180644098891207250e-01
            -5.70032258213782783151657440612325e-01 -8.21622312620400530036590680538211e-01
            -9.52013107532729829785012043430470e-01 3.06057254587887683783264947123826e-01
            4.07683614941688809985009811498458e-01 9.13123250228618843493677559308708e-01
            -3.50126449191390465287554434326012e-01 9.36702444523675126220041420310736e-01
            9.98065597133594328838057663233485e-01 -6.21696374314805247363047158160043e-02
            4.66403903874177058752614755121613e-02 -9.98911744842610804795413059764542e-01
            -5.95308864276666072079535751981894e-01 8.03496954638675653370682994136587e-01
            6.44110411450397823962532584118890e-01 7.64932531574647578409553716483060e-01
            -2.91214712227252259246768062439514e-01 -9.56657719031419606636745811556466e-01
            -8.12657913828250144661069498397410e-01 5.82741036046301008610726057668217e-01
            8.63421704896663477768470329465345e-01 5.04482863448639795578287703392562e-01
            2.61321923212860740548535432026256e-01 9.65251704193435644540954854164738e-01
            -9.52013107532729940807314505946124e-01 -3.06057254587887295205206328319036e-01
            7.70513242775789253258267308410723e-01 0.00000000000000000000000000000000e+00
            -1.08652150085474491492121273950033e-01 -9.94079830940052588950095469044754e-01
            9.68583161128631076053352444432676e-01 0.00000000000000000000000000000000e+00
            -7.28968627421411330402634121128358e-01 -0.00000000000000000000000000000000e+00
            4.81520641318652353746188055083621e-01 1.49824729125512413202558903435602e-02
            -4.66915178800144714532649459215463e-01 -1.18645768220070710330382723896037e-01
            -9.87930439740756671795907095656730e-01 1.54898180214084690842213376527070e-01
            9.29326736537534170423668911098503e-01 2.89159206528575886407939066202744e-02
            1.70243555722398520746452277307981e-01 -9.85402015288681321081298847275320e-01
            -8.91006524188367787786546614370309e-01 -0.00000000000000000000000000000000e+00
            5.62083377852130694485310868913075e-01 0.00000000000000000000000000000000e+00
            -2.10498838597022175367357021968928e-01 4.02240739859935525490897134659463e-01
            3.45131617380069588563173965667374e-01 -4.09871346768813815675258638293599e-01
            -8.11459033367125659452767649781890e-01 -2.33289268826120610444974090569303e-01
            6.88156947293836140921996502584079e-01 -3.46598720690210637673800420088810e-01
            -3.75240196866717501400501078023808e-01 4.18488372972401989624557927527349e-01
            8.56320478103113513768107623036485e-01 -2.46186646525540758778660688221862e-01
            -6.17665837467049172460065165068954e-01 -3.87148773717268968042048982169945e-01
            1.32208712735558592266116306745971e-01 4.34313515942769179378757371523534e-01
            -9.39577682361783184106229782628361e-01 1.47316923654866260839924052561400e-01
            9.42034263469989086026146196672926e-01 -3.35516685797525027190602031623712e-01
            5.19451354267116416352223495778162e-01 5.11434795760628113647783266060287e-01
            -7.41905375658127752558357315137982e-01 4.03070787023952714633878713357262e-01
            -2.74145336308877600461908485840468e-02 5.87145591960228641070784760813694e-01
            7.80411212566091694320391525252489e-01 -3.63091140558373159930027895825333e-01
            -9.78013660706298804115021994221024e-01 -9.15288064818417651524740108470723e-02
            4.10571428305382735146622508182190e-01 4.87586346164877448838836926370277e-01
            -4.81142249428901436214545128677855e-01 4.18104626285854985567880248709116e-01
            9.11900699636647549262136180914240e-01 -2.31719087237299919967981054469419e-01
            1.02845977502097028766225861318162e-02 -6.61231888422635027957596776104765e-01
            4.86650810256182231317012565341429e-01 -5.42739577183807164750817264575744e-01
            -4.27587506932606886689285374814062e-01 -6.16307991751940398295062095712638e-01
            7.70004896217444501083093655324774e-01 4.18337014009830032712500269553857e-01
            -7.29319376655605977610719037329545e-01 -4.57132457702946470856630867274362e-01
            2.47607924002176543742947956161515e-01 -2.31727415167176202526189143782176e-02
            -8.65755853912873574174113855406176e-01 -3.39044414680874484080419506426551e-01
            9.30715679095851955437979086127598e-01 2.36500079146191061196802252197813e-01
            -2.17721264485097848462302749794617e-01 -1.35618862257509704277724793541893e-02
            -6.40869261123383715528234461089596e-01 5.22827759868876085214139948220691e-01
            5.96427128828096675405845417117234e-01 -5.18285272393785412248234933940694e-01
            -9.51056516295153531181938433292089e-01 -0.00000000000000000000000000000000e+00
            8.44327925502015075309714120521676e-01 0.00000000000000000000000000000000e+00
            1.24787952277148478374968476600770e-01 1.16784588949319437234386498403182e-02
            -3.14125624175276785265964463178534e-01 -7.03573556587621440350233115168521e-01
            9.70431475564132339783895986329298e-01 1.52154507585370191158347097371006e-01
            -8.08625658528685731241125722590368e-01 2.51603170990181597421742054621063e-02
            -3.13536654749128063102148189500440e-01 7.65347661178281235550002747913823e-01
            2.35597651203856589541274502153101e-01 -7.73952352496392048664120011380874e-01
            6.34916961087438669331106666504638e-01 6.25118451711136557946701941546053e-01
            -5.74621169013445065587575300014578e-01 -6.40848928510050375173534575878875e-01
            -4.90994080973322089000987489271211e-01 8.71162908099950561435775853169616e-01
            3.68029386809962166804410799159086e-01 -7.59897372167333662318355891329702e-01
            8.35687083989954060392335577489575e-01 -4.54021579707221623056767612069962e-01
            -8.94171763287961884891785757645266e-01 3.50172558161693403544489910927950e-01
            1.31308290965072935879698334815657e-02 8.44225815235340126818641692807432e-01
            -1.62456334530964335849390067778586e-01 7.92537971682264030803821697190870e-01
            9.01890459132705490929993175086565e-01 -3.53195328037234024254331643533078e-01
            -8.02789198618462540579798769613262e-01 -4.69056303968118870617587390370318e-01
            5.07902461066308874571006981568644e-01 7.32070840963511715315803485282231e-01
            -7.20443918107661396277308085700497e-01 5.87745586603921066348732438200386e-01
            7.28827173511085280033228173124371e-01 -5.57749499431587469899795905803330e-01
            -9.44047731799498635041345551144332e-01 2.71407669435386134093590726479306e-01
            -1.17796696706233078311143458449806e-01 -8.36070322417699296480009252263699e-01
            2.70882156675796514910814494214719e-01 7.99695256317808578216954629169777e-01
            5.70032258213782561107052515581017e-01 -8.21622312620400530036590680538211e-01
            -5.61001411329723698173665979993530e-01 7.09922116404611780460243153356714e-01
            -3.66522970851886398246932685651700e-01 3.43015761620598658154790427943226e-02
            7.89745902061648297198814816511003e-01 5.29915000131442770481271509197541e-01
            -8.05844171502799855311138799152104e-01 5.05097682079852616787718488922110e-01
            1.43741388233179556710439328526263e-01 8.32002439354197265153345597354928e-01
            4.01453441623211698674467129421828e-02 -8.59805320053875110630769995623268e-01
            8.70671973886632089012493906921009e-01 4.05085517686529172909359886034508e-01
            -7.01816619833756649704525898414431e-01 -6.09866990282164622705352030607173e-01
            -4.62636303666102810527860356160090e-01 -4.02023266674627011152409750138759e-01
            6.84215978371582655981342213635799e-01 -2.12893207115975695298271119781930e-02
            -9.25348246756160897064091841457412e-01 -3.29573762865652042375330665890942e-01
            4.76889760656918126890957410068950e-01 -7.35179946455346122746732362429611e-01
            -1.44126732539939655941552132389916e-02 -3.08680705037097657950084794720169e-01
            -6.05509535035596679897196281672223e-01 -9.49381872512224250781898149398330e-02
            5.24959835115549333650619701074902e-01 -2.64402485479961757430800162183004e-01
            -3.47910136167665140494165143536520e-01 -8.49253843156532628455579470028169e-01
            2.59867316391299729705366416965262e-01 -5.82046665293017406739295438455883e-01
            7.65817447207118973118156191048911e-01 1.94598512675711543540302272958797e-01
            -7.20909951093238854369360524287913e-01 -2.07256988294474642220421856109169e-01
            -1.76149655129986926205276631662855e-01 -2.53895257557349374177846357270028e-01
            6.42125723958163918680952519935090e-01 4.60455750548987463588446189532988e-01
            -8.57531617761826514545475674822228e-01 -4.65889526359837990998613577176002e-01
            1.41689598461817500840709271869855e-01 -5.96304548117236543625097056064988e-01
            9.10660265288250281301429822633509e-01 -1.13892204252431072930562550027389e-01
            -4.73990891637744216069449976203032e-01 1.85622729194456381751621165676625e-01
            -8.93906187832087195666019852069439e-01 -1.40156063835460503730345749318076e-01
            -4.80577417671124336451260461444690e-03 3.08979622867089354087966057704762e-01
            -2.70764591773612728342612854248728e-01 8.89477852198903629066251141921384e-01
            5.16483021865079883205851274396991e-01 2.80600767884982560396167627914110e-01
            2.39678047503054658662335896224249e-01 6.41216947515009794145157684397418e-01
            -5.36332389436897538637083471257938e-01 -7.73048633288231479276930713240290e-01
            -7.40517886824198501294347352086334e-01 2.12894144086411885030685198216815e-01
            9.25348246756160897064091841457412e-01 3.29573762865652042375330665890942e-01
            1.12584083394110523457243289158214e-01 6.51658044824243609660641141090309e-01
            4.00042461626852918765706590420450e-01 2.68426212218608573056854993410525e-01
            6.29340462162034763871076847863151e-01 -2.69323825386874937670711460668826e-01
            -8.45795455360354497820196684187977e-01 1.59708749737226929354960702767130e-01
            7.10996730239270013917973756178981e-01 -6.57751464925011242179664350260282e-01
            -4.50612088996721560363312164554372e-01 -7.99513788633038346986836586438585e-01
            -1.60178887617591941827654977714701e-01 -9.27145805873857864476406120957108e-01
            2.17509355715542362919023844369804e-01 -9.15394069002036014914835959643824e-01
            -1.29718081570789040002367187298660e-01 9.20683188194668877812887330946978e-01
            4.35884418475371071544088863447541e-01 -9.00002652068530029083603949402459e-01
            -6.29407335104375853518376970896497e-01 3.67752180525778349373666742394562e-01
            -4.89903651646914850559255683037918e-01 6.19951091365638085051159578142688e-01
            7.46780663768384145306811205955455e-01 -1.89761159130402173866869475205021e-01
            3.83581673116374533005767943905084e-01 8.59140105825084487989329318224918e-01
            -8.48128132722420713385247381665977e-01 4.95547084976700857872344840870937e-01
            -3.23468614858521907873267764443881e-01 -1.75737706514007135938371106931299e-01
            8.32306432034965348876198731886689e-01 -5.21684048191385030257549715315690e-01
            -7.36644358615534566858684684120817e-01 6.40131717034015834677518341777613e-01
            1.47906520670461495009506691644674e-02 -9.50941498620659242391184307052754e-01
            5.71671343392394781801613135030493e-01 7.71593052000173362792168063606368e-01
            -2.88255537847462939282650040695444e-01 5.95182432635673275278520577558083e-01
            1.90978875157586880373372650865349e-01 9.31684263273248824965833136957372e-01
            -5.52123646605997664416065617842833e-01 7.95809536811684092860730288521154e-01
            6.90196828563563347103126943693496e-01 6.79545199278702050094125297619030e-01
            1.46324007582511426972526891177040e-02 9.40766981901440146884851856157184e-01
            3.32990041030755912210281621810282e-01 -8.90856963693840819829006250074599e-01
            -1.66572862903142326906902326300042e-01 6.15274592398701858364518102462171e-01
            8.15679528628174721482935183303198e-01 5.47316315781874451751320975745330e-01
            -6.73792470854062353069480195699725e-01 -7.05986708411275420971264793479349e-01
            1.92621045170939197976878176632454e-01 2.01824602579189049667007793686935e-01
            -2.95550258459056580662149826821405e-01 -5.64764895714953696348459288856247e-01
            4.19316529587494879471876174648060e-01 6.46424245606213254689009772846475e-01
            -6.26872567858446694266660870198393e-01 -5.11409112994704928567557544738520e-01
            -2.24691794306767639310962181298237e-02 -4.81229403181283044155236439110013e-01
            6.26355280389529367468526288575958e-01 1.18272589534903327379566917443299e-01
            -4.18577054746865129253308168699732e-01 8.64266863873912161153612032649107e-01
            -1.63466345017260733518682513931708e-01 -6.87952581248662653479186701588333e-01
            7.43772232366882724008405602944549e-01 3.18294460441047477594622705510119e-01
            -7.80075218897435140874563330726232e-01 -5.96968086086889893060458689433290e-01
            -3.29797363201309312596976042186725e-02 7.06337268585096489381669471185887e-01
            3.29786835763509866836784567567520e-01 -2.21285087596854335156848492260906e-01
            -2.69005891187314161161481251838268e-01 2.05862111906548628237345610614284e-01
            -5.33495599045166968288356201810529e-01 4.99279482544807695632016475428827e-02
            5.12422198857500621471672275220044e-01 -4.18039320322732055323911026789574e-01
            6.24710712177461990890492415928748e-01 -6.54559791741803231879259783454472e-01
            -1.35338076804238371941835339384852e-01 2.08639078182871223754446532439033e-01
            1.41761257975513005336054561666970e-01 -2.04329160217694566492596663920267e-01
            -7.54962930158121880985788720863638e-01 4.73205663779819374337165527322213e-01
            3.14131470184790262401008931192337e-02 1.53248021347156898608332653566322e-01
            -3.67178644538921683526666583929909e-01 -8.96288560128093769741042251553154e-01
            2.85857854033248148795109955244698e-01 -8.43906341624319433769585430127336e-01
            5.71032987105135059202609681960894e-01 5.98317310446660699518872661428759e-01
            -5.77934866838112237807933979638619e-01 -2.68887769152766187286118793053902e-01
            -1.36155357822940165091196718094579e-01 9.66372208193673642995236150454730e-01
            4.30005662760491069729340551930363e-01 -6.62902763289110352751265509141376e-01
            -7.84047013277229476457819146162365e-01 -9.80572514069376760525997838158219e-02
            1.78124417605785761997694294223038e-01 7.49641480406108451362001687812153e-01
            -2.84201318985743855982661898451624e-01 9.33618303440851171615122439106926e-01
            1.19828770337885668478827838612233e-02 -7.70420059416196179036262492445530e-01
            4.89224912137107337262165174252004e-01 1.40649025213763351205287222001061e-01
            -5.05378129101968553982260345946997e-01 -8.34869135169341025637379516410874e-01
            -1.49076591494904792112308200557891e-01 -3.98828503236562159628419976797886e-01
            6.36180343445494678178420144831762e-01 -7.55514942778201525896974999341182e-01
            4.55171387624476730948686054034624e-02 9.74854715489385070803507460368564e-01
            3.72373705991493109124235161289107e-01 9.08969782792743208332808535487857e-01
            -4.31318276997815219786502893839497e-01 -2.70347381856710977654500993594411e-01
            -5.74621169013445065587575300014578e-01 6.40848928510050264151232113363221e-01
            5.08677084332258666599102525651688e-01 8.40318907807838022527846533193951e-01
            -4.37234899512399521892547227253090e-01 -7.22298614446212017270454452955164e-01
            -1.37044138725707392856278943327197e-01 -9.72680393030030621126513779017841e-01
            2.05919897797178813592111623620440e-01 -6.76459160401571724108293892641086e-01
            -2.84358408089210645819377987209009e-01 6.94123124907333854949342821782921e-01
            4.31103449403852589938423989224248e-01 -8.23792528263522005893548794119852e-01
            -4.98039087624729870373840867614490e-01 3.12167535164137643910464703367325e-01
            -4.93270439150807551986765986384853e-02 4.51302799229478746045884918203228e-01
            3.22482940091893988121540814972832e-01 1.03673197895084640007823395535524e-01
            5.19322793438008356581292446207954e-01 -1.31962837318722520096869743611023e-01
            -2.70882156675796459399663262956892e-01 -7.99695256317808356172349704138469e-01
            1.37044138725707281833976480811543e-01 -9.72680393030030621126513779017841e-01
            -1.87607133107599216570449129903864e-01 5.01910268698112171925629354518605e-01
            3.61639442800233620878280760280177e-02 -1.52196948019776540084180282974557e-01
            1.63686949128930142594029462088656e-01 5.37721402161494888005677239561919e-01
            -1.31174914181346136121675272079301e-01 -7.59265302237879580182777772279223e-01
            -1.41439714617997730483622831343382e-01 -1.22908763660910899662148665356654e-01
            ];
        
end



stats_matrix=[
    1    3 3.000232e+00 1.43652e+00 4.047e+00   0
    2    6 3.041392e+00 2.84732e+01 8.275e+00   0
    3   10 4.038372e+00 1.99252e+03 2.541e+01   0
    4   15 5.773425e+00 3.51034e+06 3.056e+01   0
    5   21 9.257063e+00 1.12952e+10 3.940e+01   0
    6   28 9.874270e+00 2.14364e+15 7.207e+01   0
    7   36 1.107126e+01 5.80582e+21 7.795e+01   0
    8   45 1.692800e+01 4.26418e+28 1.559e+02   0
    9   55 1.945967e+01 1.13045e+37 1.573e+02   0
    10   66 1.900962e+01 1.12578e+47 1.246e+02   0
    11   78 2.512112e+01 2.94460e+57 2.275e+02   0
    12   91 3.573396e+01 1.09277e+71 2.948e+02   0
    13  105 3.731040e+01 6.56673e+84 2.198e+02   0
    14  120 2.974765e+01 9.55787e+100 3.211e+02   0
    15  136 5.673142e+01 2.44186e+116 7.479e+02   0
    16  153 5.647200e+01 1.14228e+135 5.604e+02   0
    17  171 5.800516e+01 1.03123e+154 6.874e+02   0
    18  190 7.475907e+01 1.11798e+176 1.367e+03   0
    19  210 1.483895e+02 2.54107e+198 2.184e+03   0
    20  231 1.061188e+02 4.68313e+222 1.498e+03   0
    ];


