
function demo_RBF_cub_OPT_basic

%--------------------------------------------------------------------------
% * Object:
%--------------------------------------------------------------------------
% In this "basic" demo we test routines on RBF cubature over polygons via
% triangulation ("polyshape" version).
%
% With respect to the basic version, an "almost" optimal shape parameter is
% determined.
%
% Note that the battery of tests is used in the reference paper mentioned
% below.
%--------------------------------------------------------------------------
% Reference papers:
% 1. R. Cavoretto, A. De Rossi, A. Sommariva, M. Vianello
% RBFCUB: a numerical package for near-optimal meshless cubature on general
% polygons
%
% 2. A. Sommariva and M. Vianello,
% RBF moment computation and meshless cubature on general polygonal regions.
%--------------------------------------------------------------------------
%% Copyright (C) 2021- R. Cavoretto, A. De Rossi, A. Sommariva, M. Vianello
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Authors:
%%          Roberto Cavoretto <roberto.cavoretto@unito.it>
%%          Alessandra De Rossi   <?alessandra.derossi@unito.it>
%%          Alvise Sommariva <alvise@euler.math.unipd.it>
%%          Marco Vianello   <marcov@euler.math.unipd.it>
%%
%% Last Update: July 9, 2021.
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
% * RBF_type:
%--------------------------------------------------------------------------
% 1: phi=@(r) (1+r.*r).^(1/2);             % Multiquadric
% 2: phi=@(r) exp(-r.*r);                  % Gaussian
% 3: phi=@(r) (1+r.*r).^(-1/2);            % Inverse Multiquadric
% 4: phi=@(r) (1+4*r).*(max(0,(1-r))).^4;  % Wendland 2
% 5: phi=@(r) r.*r.*log(r+eps*(1-sign(r))); % TPS
% 6: phi=@(r) r.^3;                        % polyharmonic spline
% 7: phi=@(r) r.^5;                        % polyharmonic spline
% 8: phi=@(r) r.^7;                        % polyharmonic spline
% 9: phi=@(r) (max(0,(1-r))).^2;             % Wendland W0
% 10: phi=@(r) (35*r.^2+18*r+3).*(max(0,(1-r))).^6;         % Wendland W4
% 11: phi=@(r) (32*r.^3+25*r.^2+8*r+1).*(max(0,(1-r))).^8;  % Wendland W6
% 12: phi=@(r) (sqrt(2)/(3*sqrt(pi))*(3*r^2*log(r/(1+sqrt(1-r.^2)))+...
%            (2*r^2+1).*sqrt(1-r^2)))*max(0,(1-r));  % Missing Wendland
% 13 phi=@(r) exp(-r);                     % Matern beta_1=3/2.
% 14 phi=@(r) (1+r).*exp(-r);              % Matern beta_2=5/2.
%--------------------------------------------------------------------------
% Note: In the tests of the reference paper: "RBF_typeV=[4 5 6]".
%    In other words, we make experiments by means of Wendland 2, TPS and
%    r^3 polyharmonic spline.
%--------------------------------------------------------------------------

RBF_type=13;

%--------------------------------------------------------------------------
% Define polygon (last vertex may not be repeated).
%--------------------------------------------------------------------------

polygon_sides=(1/4)*[1 0; 3 2; 3 0; 4 2; 3 3; 3 4; 2 4; 0 3; 1 2];

%--------------------------------------------------------------------------
% Define function.
%--------------------------------------------------------------------------

f=@(x,y) franke(x,y);

%--------------------------------------------------------------------------
% Define centers.
%--------------------------------------------------------------------------
% We load 200 centers from an attached routine of points in polygon.
Ncenters=200; centers=list_of_points(Ncenters); 








% ...................... main code below ..................................

% polygon in polyshape form
P=polyshape(polygon_sides(:,1),polygon_sides(:,2));

% ... function values at centers ...
fcub=feval(f,centers(:,1),centers(:,2));

% Choose optimal shape parameter
RBF_scale=RBF_optimal_scale(centers,fcub,RBF_type);

% .... RBF cubature weights ....
[weights,cpus,Vcond]=RBF_cub_polygon_OPT(P,centers,RBF_type,...
    RBF_scale);

% .... integration by scattered data ....
cubature_value=weights'*fcub;








% ...................... statistics below .................................

switch RBF_type
    case 1, RBF_string=' MULTIQUADRIC: (1+r^2)^(1/2)';
    case 2, RBF_string=' GAUSSIAN: exp(-r^2)';
    case 3, RBF_string=' INVERSE MULTIQUADRIC: (1+r^2)^(-1/2)';
    case 4, RBF_string=' WENDLAND, W2: (1+4*r)*(max(0,(1-r))^4)';
    case 5, RBF_string=' TPS: r^2*log(r))';
    case 6, RBF_string=' POLYHARMONIC: r^3';
    case 7, RBF_string=' POLYHARMONIC: r^5';
    case 8, RBF_string=' POLYHARMONIC: r^7';
    case 9, RBF_string=' WENDLAND, W0: max(0,(1-r)^2)';
    case 10, RBF_string=' WENDLAND, W4:(35*r^2+18*r+3)*max(0,(1-r))^6';
    case 11, RBF_string=' WENDLAND, W6:(32*r^3+25*r^2+8*r+1)*max(0,(1-r))^8';
    case 12, RBF_string=' MISSING WENDLAND:';
    case 13, RBF_string=' MATERN B1=(d+1)/2: exp(-r)';
    case 14, RBF_string=' MATERN B2=(d+3)/2: (1+r)*exp(-r)';
end
fprintf('\n \t RBF TYPE     : %2.0f',RBF_type);

fprintf('\n \t RBF SCALE    : OPTIMAL, %1.1e',RBF_scale(1));

% ... experiment data ...
fprintf('\n \t NODES CARDIN.: %4.0f',Ncenters);

% ... cubature values ...

fprintf('\n \n \t * CUBATURE RES.: %1.15e',cubature_value);

% Approximation of the required integral by algebraic rule.
[xyw,xvc,yvc,pgon,tri]=polygauss_2018(100,P);
fxy=feval(f,xyw(:,1),xyw(:,2)); IR=(xyw(:,3))'*fxy;

fprintf('\n \t * EXACT VALUE  : %1.15e',IR);

% ... cubature errors ...

ae=abs(IR-cubature_value);
fprintf('\n \t ** ABSOLUTE ERR.: %1.3e',ae);
if abs(IR) > 0
    re=ae/abs(IR);
    fprintf('\n \t ** RELATIVE ERR.: %1.3e',re);
else
    re=NaN;
end

fprintf('\n \n');







% ............................... plots ...................................

    clf;
    hold on;
    axis square;
    % plot(xv,yv,'k-','LineWidth',4);
    plot(P);
    plot(centers(:,1),centers(:,2),'r.');
    hold off;

% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
% MAIN PROGRAM ENDS HERE.
% +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++




function centers=list_of_points(Npts)

%------------------------------------------------------------------
% Object:
%------------------------------------------------------------------
% Set of random points inside the domain considered in the example.
%------------------------------------------------------------------
% Input:
%------------------------------------------------------------------
% Npts: number of centers (choose below 4998).
%------------------------------------------------------------------
% Output:
%------------------------------------------------------------------
% centers: Npts x 2 matrix with the cartesian coordinates of the
%     centers.
%------------------------------------------------------------------

pts=[7.272331899610127e-01     5.283869139225909e-01
    7.771546724148052e-01     5.009953509614262e-01
    7.540415170682531e-01     1.107329430275575e-01
    3.396016609554527e-01     7.555669209652783e-01
    2.883394725494742e-01     7.185016250281264e-01
    5.054673849963799e-01     4.384083386904444e-01
    7.826222229548754e-01     5.494515575256100e-01
    6.696782200564841e-01     6.466024302146121e-01
    9.427661566506995e-01     5.418770123616523e-01
    9.153334543916269e-01     4.517671467593434e-01
    8.272732902576944e-01     6.050474047534719e-01
    7.107954979232102e-01     8.167319254314060e-01
    4.480505516133119e-01     5.276169020210121e-01
    2.531055121323472e-01     6.882577212140822e-01
    7.152936446557150e-01     7.484804573784315e-01
    7.852453121263325e-01     4.450255566089780e-01
    5.994340696316437e-01     9.803752065358604e-01
    4.072384605666730e-01     8.918954888148217e-01
    7.532063227304264e-01     1.521783839339380e-01
    1.747895352778353e-01     8.120425761436404e-01
    5.200022694137976e-01     8.810217304862384e-01
    8.893970057628845e-01     4.745309713504579e-01
    4.580248267889120e-01     7.892819227242356e-01
    8.215127216368449e-01     4.552421742714163e-01
    3.643942616304536e-01     5.114591939730792e-01
    3.124968748515734e-01     2.099354679222175e-01
    3.510115757984480e-01     1.298374980494479e-01
    5.694618481872386e-01     3.619959643692180e-01
    5.382703715874814e-01     5.484820588531044e-01
    1.140176815850541e-01     7.233600000953595e-01
    4.015522424893179e-01     3.558695532277149e-01
    2.669934878616841e-01     5.931912001183531e-01
    2.925424110552437e-01     3.373575760933787e-01
    5.496667448591944e-01     5.709836142768510e-01
    7.895482227584464e-01     1.605152659015101e-01
    3.614785293907666e-01     6.492747728720951e-01
    6.841711703070147e-01     4.460538131859101e-01
    8.110291833751577e-01     2.113827660112769e-01
    3.336310117930497e-01     1.401030719012717e-01
    3.618530120897696e-01     2.437163181564611e-01
    5.674613551012467e-01     8.524708040216021e-01
    3.006270715957233e-01     7.552822836088301e-01
    5.633590144452877e-01     3.613446711426180e-01
    2.945205956924253e-01     1.005195293529485e-01
    7.882511828382536e-01     1.872888257861151e-01
    8.629921088018315e-01     5.174636983897900e-01
    7.165090329112872e-01     6.907480213537980e-01
    2.576329767890394e-01     4.386874937364053e-01
    3.025956525254114e-01     5.424450916343149e-02
    3.221232961282005e-01     8.403434132513448e-01
    2.772653687796230e-01     1.922022150044623e-01
    3.800398949231472e-01     7.081915668193516e-01
    8.464470637995358e-02     7.405558348859249e-01
    5.983090242193976e-01     4.148957137233071e-01
    6.597895908812729e-01     8.664881019797690e-01
    2.802759623770258e-01     2.680126526760104e-01
    8.571685703862187e-01     4.200068222665099e-01
    3.232609880268509e-01     1.109034405742689e-01
    3.913725420406216e-01     2.789085347408381e-01
    7.974041016033540e-02     7.107349756729980e-01
    2.718786416181702e-01     3.638905264412307e-01
    8.201770641514223e-01     1.874160882144223e-01
    6.845495272377272e-01     5.000202376751590e-01
    6.501354061688231e-01     9.385678350828126e-01
    5.635752518120565e-01     6.143068621917606e-01
    4.760810383237628e-01     2.616280963690372e-01
    3.094452622649160e-01     4.908480069076123e-01
    4.602600703487966e-01     4.835972391138028e-01
    6.697945479524802e-01     5.153891794560927e-01
    7.493453507361294e-01     9.898562083149608e-01
    9.022702691478126e-01     4.027879481583997e-01
    1.715489044248474e-01     7.569533557803514e-01
    5.860549739395788e-01     4.966332263804520e-01
    5.360526731997248e-01     5.873739927751538e-01
    4.864095544289324e-01     6.485235645014089e-01
    1.090755034182084e-01     7.730362335223798e-01
    5.649985513754906e-01     3.590323156014049e-01
    2.206905446530958e-01     6.609044196631053e-01
    6.997501127740463e-01     4.929743837150021e-01
    6.807059773876385e-01     7.485158964751628e-01
    4.392424279861942e-01     7.135913739267719e-01
    6.429273772887396e-01     4.472051045122715e-01
    3.431184473442960e-01     3.579761104491374e-01
    8.866785667616800e-01     6.113079164711561e-01
    3.497478379287583e-01     7.731973112569447e-01
    3.147422700864182e-01     2.228306746084194e-01
    6.061742298805880e-01     9.445880033987581e-01
    1.152942361026310e-01     6.990177591584144e-01
    7.341102134840928e-01     8.116439623171432e-01
    5.421176177587754e-01     7.202179982152304e-01
    3.036264022916508e-01     5.753964293288104e-01
    9.695600145486165e-01     4.480699499726556e-01
    1.746582241225253e-01     8.106889391113896e-01
    5.427174369474771e-01     3.281928817698714e-01
    3.134331074208520e-01     7.226769135819526e-01
    6.727136171791750e-01     4.654464623856626e-01
    4.948148771075445e-01     6.009126163542458e-01
    5.251379932051473e-01     9.217029644322928e-01
    3.721500022581837e-01     7.292472248706113e-01
    6.498063437996823e-01     9.305733319434006e-01
    3.267674850798618e-01     7.806481982965704e-01
    7.881677429695485e-01     3.817341049621202e-01
    6.645337355407702e-01     9.406149852841695e-01
    5.717401768436763e-01     8.554212037430728e-01
    8.359135464938557e-01     5.205816473121123e-01
    4.794441725759290e-01     4.369212200716895e-01
    4.598208866253480e-01     9.459988284243636e-01
    4.591217010285725e-01     3.706014150444543e-01
    3.041155455165464e-01     1.549438195005989e-01
    3.828694377292954e-01     3.668247822849339e-01
    6.347832821894142e-01     5.032286737352492e-01
    6.617366250275847e-01     7.524096125754463e-01
    5.497962474096100e-01     3.678469886117837e-01
    6.836754037146654e-01     5.296392976943165e-01
    7.568130822199575e-01     3.791245865523986e-01
    9.431419479360348e-02     6.950914113199533e-01
    7.258734674868385e-01     7.844616642503963e-01
    5.872787704967799e-01     4.615556477778691e-01
    4.663727139588518e-01     3.574176577630042e-01
    4.420934898249446e-01     8.559565549497734e-01
    3.258705300152881e-01     1.715185340979862e-01
    3.518534889382556e-01     9.099204846265783e-01
    7.844452683587954e-01     9.098182845722291e-02
    6.269585958962889e-01     5.772026844969409e-01
    3.775333798091398e-01     7.110972433010895e-01
    6.886602953246259e-01     8.221456383878177e-01
    3.389597625130424e-01     8.876375266801776e-01
    3.600820759740054e-01     6.435130413024917e-01
    3.733579507299569e-01     8.955175277517888e-01
    2.994997357282190e-01     4.865402919516598e-01
    4.026887850026639e-01     1.735286578384104e-01
    5.496716301119351e-01     4.690400184502006e-01
    1.023184418484259e-01     6.606600144075065e-01
    3.560288544793985e-01     4.816503831904250e-01
    3.553792907388242e-01     2.062987953460228e-01
    5.457457360487445e-01     8.567584415233030e-01
    3.540388278160987e-01     8.049328906155931e-01
    3.100542892546668e-01     1.351158898207966e-01
    5.670731287680973e-01     8.064830005860302e-01
    9.002916642536785e-01     5.856905296360080e-01
    4.859971105731972e-01     4.069027643584200e-01
    6.347183708840002e-01     6.975958760165886e-01
    2.678481868447209e-01     1.782715889797268e-01
    2.369750193874933e-01     6.771544131463436e-01
    3.645055290736777e-01     8.489535304191324e-01
    8.567532824539668e-01     4.499357101667052e-01
    4.843016087664992e-01     2.892838584275382e-01
    6.477691470948908e-01     4.481495812862027e-01
    8.863011831855592e-01     4.890519944714812e-01
    5.488772823770943e-01     7.637472374030604e-01
    7.749033897619083e-01     2.219744329610230e-01
    2.413107633110876e-01     6.731875315915783e-01
    3.140905099539509e-01     3.523175432922857e-01
    8.994706761533270e-01     4.988183443557348e-01
    7.537213693765544e-01     5.622632094234667e-01
    7.715061033923312e-01     1.643217305129097e-01
    8.327417976473439e-02     6.687822091474158e-01
    8.201203348076030e-01     4.657874604696589e-01
    3.456078822307246e-01     2.250154158966875e-01
    5.359582790726438e-01     8.476279544922697e-01
    3.503929497971190e-01     6.414401340894493e-01
    4.635547644156025e-01     3.445133224428255e-01
    4.938376920255959e-01     9.374589361292784e-01
    3.548882582960062e-01     5.105130100337409e-01
    3.376735822242226e-01     1.485855671073267e-01
    3.872110467735783e-01     4.423545687668053e-01
    6.849859381463231e-01     7.873374195952246e-01
    3.168787947356062e-01     2.785596013846646e-01
    7.173999582577850e-01     5.607729512621258e-01
    4.558357163008188e-01     8.172147114131924e-01
    7.223050635401824e-01     5.449251972660276e-01
    4.863569597652435e-01     2.496553552157289e-01
    3.084186523102938e-01     2.541166655922699e-01
    7.227734429941653e-01     8.680028484145079e-01
    3.441332938481586e-01     2.270490923180311e-01
    4.045070725899341e-01     8.315494737173480e-01
    4.831611444672986e-01     2.457347673183617e-01
    5.674557100711187e-01     5.435273451176275e-01
    7.538516661574231e-01     6.598221078951595e-01
    8.387428999100761e-01     2.824492356787405e-01
    5.425609406914055e-01     6.103124343861834e-01
    8.399801650852503e-01     3.977186240577877e-01
    9.046766542152456e-01     5.757733258456912e-01
    5.941617812587781e-01     6.730256712303316e-01
    3.277618909018574e-01     6.972699070446682e-01
    3.101270399897629e-01     8.344552190521571e-01
    1.054635934727034e-01     7.527521991800943e-01
    5.262201602027210e-01     5.061433252618774e-01
    7.851305821954956e-01     8.458853983225523e-02
    3.205445872396632e-01     6.461360733341037e-01
    3.401484620794153e-01     1.834321835233484e-01
    7.634876548520210e-01     3.798917299314996e-01
    2.544727578306485e-01     3.055366700966098e-01
    6.542097553997128e-01     6.145762170826214e-01
    5.800881428609036e-01     3.487388684227916e-01
    4.347137024311586e-01     2.962167851230408e-01
    2.792899557250491e-01     5.656132881064414e-01
    2.962005209358717e-01     7.263414920861876e-01
    3.915072108535130e-01     1.888790879540763e-01
    4.277108029197064e-01     7.951734786379507e-01
    5.133988958940809e-01     6.234768402112321e-01
    8.661453235096017e-01     2.919828118040815e-01
    3.531990131063084e-01     3.213873123875187e-01
    3.900082619192891e-01     9.079966190416745e-01
    5.044934463355377e-01     3.039510764428945e-01
    4.299684984709717e-01     5.220539737727137e-01
    6.624455140515367e-01     8.145482113112489e-01
    5.726153147557421e-01     8.535259808897059e-01
    5.020955137707388e-01     8.840715193084595e-01
    7.359999109665880e-01     5.892134224007782e-01
    3.660392710796982e-01     2.671495230599839e-01
    7.693195459627896e-01     5.324541802490635e-01
    4.702274785116899e-01     6.372868482125286e-01
    1.487761087254698e-01     7.485188650086969e-01
    3.860000398914653e-01     7.879820492956253e-01
    7.011509827160689e-01     8.830120610753905e-01
    5.166408722221416e-01     6.496187156005024e-01
    7.857854986379914e-01     4.232933076646211e-01
    4.379824254730770e-01     4.037028233700832e-01
    4.220303129797623e-01     6.337830042913813e-01
    7.519371406501685e-01     5.022344234332440e-01
    2.904994694208506e-01     6.543677315484620e-01
    3.698243381544720e-01     5.276082801340858e-01
    8.347917862800016e-01     4.663935013590014e-01
    5.171754808099434e-01     9.246365493040519e-01
    4.728868015890362e-01     2.626393985073625e-01
    4.157091730969636e-01     5.722755748699727e-01
    3.806471297985997e-01     3.079050200788702e-01
    3.009980023543426e-01     4.768400839026100e-01
    4.108821561422890e-01     4.490172193987656e-01
    9.651106216882648e-01     4.451513792297523e-01
    3.996621053605226e-01     6.170759300913807e-01
    5.668856500819951e-01     5.750614439246137e-01
    8.039854128379732e-01     3.523763749214621e-01
    5.510200623152489e-01     7.074074040004648e-01
    9.214071872346306e-01     4.971345227121537e-01
    5.919698899784231e-01     9.230318171159502e-01
    5.934081443331819e-01     5.176137504061707e-01
    4.831094336680907e-01     3.689728354159102e-01
    3.861325855112661e-01     6.314521535446316e-01
    2.806380827419408e-01     7.682705295395692e-01
    2.154531386977033e-01     6.695347675073950e-01
    1.434263010813948e-01     6.946071870169213e-01
    6.521610223504231e-01     7.953438787037170e-01
    7.506860393335507e-01     4.840068427916837e-01
    5.105889922071767e-01     4.459207211132469e-01
    6.636964615609673e-01     9.816345612115452e-01
    3.981077887005652e-01     2.984350265527712e-01
    7.883826310355820e-01     4.209682602776814e-01
    7.463009733930127e-01     7.593525946147466e-01
    7.420766662100555e-01     9.052644739392445e-01
    3.535733765961548e-01     5.848394860293921e-01
    8.630143070890541e-01     5.393112243144076e-01
    4.218720432254363e-01     5.799069929564693e-01
    5.164206108248762e-01     5.251460912727244e-01
    6.511899944883668e-01     7.402766968657828e-01
    7.087243542534982e-01     8.484145907203738e-01
    6.380036963549175e-01     7.948420873697141e-01
    6.665305677777962e-01     4.361870969595661e-01
    6.027881921892682e-01     9.077767522794609e-01
    1.503380457884147e-01     7.480714929063497e-01
    6.209063563937265e-01     4.223166356955832e-01
    6.392316736646863e-01     9.958781934625347e-01
    5.889234604576435e-01     5.689769643698827e-01
    1.586574611086069e-01     6.432375272372178e-01
    4.810902938841086e-01     9.510963496295576e-01
    1.385268847219576e-01     6.776786258726590e-01
    2.749855183028360e-01     9.634840671182554e-02
    7.565738659392918e-01     6.330777248037526e-01
    1.581745273371661e-01     6.766954667656608e-01
    4.064518100645534e-01     3.403157214811442e-01
    4.813928741894947e-01     4.454525050379692e-01
    9.200865837155237e-02     7.926501218997173e-01
    3.290297433803379e-01     2.135978437030630e-01
    7.843155833422844e-02     7.600560356405016e-01
    2.521203393221845e-01     6.321329497437856e-01
    4.961168739714836e-01     8.348999872466744e-01
    7.183089960264323e-01     9.816165893486619e-01
    7.738549274693057e-01     5.992797825689850e-01
    8.631466706986201e-01     6.324290401646173e-01
    6.043903497982172e-01     6.345972866819687e-01
    6.185171686605028e-01     7.680862533158492e-01
    6.946593183395341e-01     7.562991359424226e-01
    3.763679163403241e-01     8.216480754582884e-01
    7.458737316724849e-01     6.886975594515853e-01
    4.061027769564188e-01     1.816773909725664e-01
    2.588754699115461e-01     8.071899127100041e-01
    9.351951617504615e-01     4.704003964381924e-01
    5.355493348664085e-01     9.022145325466580e-01
    6.516382697434530e-01     8.735123689685603e-01
    2.060615033502277e-01     8.455427046654552e-01
    2.513361544552947e-01     4.904521662579756e-01
    3.364856565760184e-01     6.227379306787998e-01
    4.517673738690381e-01     7.728976980042002e-01
    1.692863648077991e-01     7.535827380147950e-01
    5.367777990042312e-01     9.255512893587807e-01
    4.235342237889510e-01     8.256122692589318e-01
    3.949703485182953e-01     6.916742533153968e-01
    3.459686599182676e-01     4.734800963343786e-01
    7.744139832812664e-01     4.510302181179133e-01
    3.220120878294338e-01     2.653322879761099e-01
    6.458315313312377e-01     8.069347123311060e-01
    8.258850483457415e-01     4.047080667640282e-01
    2.724912064911779e-01     1.697830791857086e-01
    2.660056192354570e-01     1.268687456459302e-01
    4.001822007443511e-01     9.141692080715026e-01
    2.650735266519587e-01     6.536728894324969e-01
    7.147374746651307e-01     6.633285323945161e-01
    3.074246046085640e-01     6.558361065765141e-01
    3.377328229056848e-01     7.023950746497077e-01
    5.609899634208215e-01     8.999738042477116e-01
    4.284346015795227e-01     5.256061374721710e-01
    7.393641242670722e-01     6.408875581590353e-01
    3.942201052710618e-01     7.700252952258082e-01
    1.333582709896223e-01     7.860555967018954e-01
    5.615429050533837e-01     6.791059079753481e-01
    8.063471526840007e-01     3.466592795696547e-01
    9.897561271647738e-01     4.947122335486098e-01
    8.987438093472856e-01     3.232269306921998e-01
    9.311836551817478e-01     4.892763095979970e-01
    8.843810017422400e-01     5.877112477807738e-01
    3.348817056369809e-01     3.471145578959449e-01
    3.000241127260921e-01     8.991318397243897e-02
    9.755365765971101e-01     5.230808545501577e-01
    7.696629902817186e-01     1.409947348249426e-01
    3.547781972938806e-01     4.219096817722188e-01
    4.620643503607964e-01     3.693471562553110e-01
    5.504748528146649e-01     8.684027969702813e-01
    5.689806042552973e-01     6.047923107484501e-01
    8.400126112087182e-01     3.414574712541768e-01
    2.681208518839848e-01     8.010698721935839e-01
    5.341151233829952e-01     7.830924012207648e-01
    7.291898483733250e-01     5.004376463361533e-01
    1.689573811264126e-01     6.388053331642638e-01
    4.944366398125023e-01     5.468101056129767e-01
    2.552186111182128e-01     1.167325936667462e-01
    6.208784131336947e-01     4.068678428654907e-01
    8.234902549948452e-01     2.458012051083183e-01
    4.747899355298033e-01     8.883395767244290e-01
    8.005511697071706e-01     3.564728567265635e-01
    7.749348225817280e-01     3.404761265761270e-01
    8.623435868978699e-01     4.684582746704541e-01
    6.987314563212913e-01     9.484100744051621e-01
    5.433212727932089e-01     5.126976052817549e-01
    7.520093823676625e-01     6.208505160244286e-01
    1.318909648744998e-01     6.698627669877615e-01
    8.435700490261195e-01     3.880483081911466e-01
    7.587286326405994e-01     6.002687588709501e-01
    2.177126782417204e-01     8.569336551075758e-01
    8.783206530074954e-01     3.740194107612864e-01
    6.190036095954851e-01     5.098548946317080e-01
    4.167129667462212e-01     6.931483784073382e-01
    5.311863089062335e-01     7.126431761511671e-01
    6.025920722787267e-01     8.439757922090726e-01
    5.593358228838200e-01     8.165579998177520e-01
    6.223016867713246e-01     9.047166214969247e-01
    4.421255081991798e-01     9.259634401742777e-01
    4.628620267347966e-01     4.106346899269270e-01
    6.005467706721319e-01     7.518187223981825e-01
    6.642174975827617e-01     6.265973605359911e-01
    7.404285979891699e-01     7.336150490127642e-01
    5.115309289808684e-01     7.945717343226828e-01
    8.909242083602457e-01     3.720500280430596e-01
    3.508374808793729e-01     8.506575049239041e-01
    7.098606227304174e-01     6.825833560836689e-01
    8.099723592602003e-01     6.741060328973715e-01
    4.996494207431356e-01     4.837469862783117e-01
    5.735264951526232e-01     9.758829310163294e-01
    5.379058086639067e-01     8.730627014733804e-01
    4.400610822219665e-01     2.161630246203927e-01
    5.957440012305285e-01     3.923471123420650e-01
    8.236256837540251e-01     2.902864886003942e-01
    6.462582406304036e-01     6.137366924524397e-01
    5.586564924299813e-01     4.179146070987867e-01
    5.614523037432189e-01     5.830859868978679e-01
    8.761367900915580e-01     6.095257519089257e-01
    3.560899455125786e-01     3.924098887071363e-01
    7.444311258444307e-01     6.254516104206010e-01
    7.546996510845497e-01     3.972634442900327e-01
    4.968170974450782e-01     9.727023104967882e-01
    8.874857444570743e-01     4.494743257789936e-01
    3.108536377726800e-01     5.325677637940769e-01
    4.300049192634746e-01     6.486475905080479e-01
    4.889685997065052e-01     9.062728047459415e-01
    2.921184682561530e-01     2.435816100875114e-01
    3.792473527349337e-01     8.269514692698504e-01
    3.385251046109726e-01     1.150408094241531e-01
    2.558729143803457e-01     2.697580121365841e-01
    8.448706131994060e-01     2.380545963405041e-01
    2.245981673669168e-01     7.430683135198290e-01
    4.005107734163713e-01     1.572854517946647e-01
    8.861046979450515e-01     5.184679573713294e-01
    4.668196207869358e-01     5.910900205442498e-01
    3.461263630903569e-01     1.826592623960325e-01
    1.245200895719374e-01     7.022626206781090e-01
    8.614422273961587e-01     4.125851346853621e-01
    7.379648838070741e-01     8.303273536415159e-01
    1.397052097377773e-01     7.512792429490844e-01
    2.666328504489973e-01     8.284972245897190e-01
    4.934271393837537e-01     7.868746053703343e-01
    5.621941465892720e-01     4.084460336981206e-01
    7.321332705922541e-01     6.027294975590189e-01
    7.763306464766438e-01     5.994139555314348e-02
    7.608829879858001e-01     3.463377711616916e-01
    6.083771945282144e-01     8.594944493412378e-01
    2.616859664037160e-01     5.863753729701856e-01
    5.361273097509232e-01     7.233973226217775e-01
    3.962358979200737e-01     8.748105565099593e-01
    3.582259244728115e-01     6.407917325550113e-01
    4.442555316703395e-01     7.958111727625324e-01
    4.342537120952569e-01     4.425832455770284e-01
    8.027476873358730e-01     5.805955899531111e-01
    3.483944334891440e-01     1.212549528700171e-01
    5.803375247194755e-01     5.250011117582898e-01
    5.858721872051385e-01     7.766557649147794e-01
    8.333253676150152e-01     6.555009863994833e-01
    3.410749192641870e-01     6.420431911062597e-01
    4.765228991705449e-01     7.667988412704287e-01
    4.870569298408740e-01     8.972994139773033e-01
    9.033051344762515e-01     5.491645485071567e-01
    7.158601056145031e-01     8.256998200675643e-01
    3.469948516009970e-01     7.130032897107866e-01
    1.179819094459890e-01     7.768132923636890e-01
    5.063815976875328e-01     6.457741057676212e-01
    3.811460556846022e-01     3.205090517927482e-01
    6.660724869519832e-01     5.500630247240889e-01
    8.816445765966210e-01     5.111596458476041e-01
    5.529461957574808e-01     4.676557905257555e-01
    5.403517241489464e-01     4.973045547009253e-01
    8.451821328331819e-01     5.207877682889571e-01
    7.838145172449220e-01     3.514600999943683e-01
    2.218443120925320e-01     5.412679373729670e-01
    7.175374617954930e-01     7.240385682765589e-01
    2.194641609338784e-01     8.307124264882679e-01
    9.572241676017725e-01     4.946246131357454e-01
    7.316675471155980e-01     5.621233214142046e-01
    8.904902205219083e-01     4.664267589478601e-01
    9.001853611331235e-01     4.434899641335656e-01
    2.251963766043075e-01     5.804626802794581e-01
    4.898723096101909e-01     7.494302474016835e-01
    6.451180791000333e-01     5.208219253019939e-01
    1.042480099016944e-01     7.953289258902242e-01
    4.512467671994443e-01     4.760253057110282e-01
    5.504416603334700e-01     6.069160765680880e-01
    2.872707818976455e-01     7.948769338569547e-01
    7.030918712785731e-01     4.977935434568431e-01
    1.003014463736869e-01     7.649232184775648e-01
    4.158933016103252e-01     2.018936975106713e-01
    5.504295692253033e-01     6.754508656003853e-01
    7.571561945759120e-01     1.700280809826955e-01
    6.269818827662782e-01     8.224734305079601e-01
    7.762024519548644e-01     5.958951113877468e-01
    5.203469837047449e-01     9.973483506663277e-01
    5.158598728658617e-01     9.560332456631372e-01
    2.641902126898400e-01     2.098201809368723e-01
    5.186217845805716e-01     6.929931112020224e-01
    7.836772273814894e-01     4.623723353688042e-01
    6.918787819622145e-01     4.906458788624392e-01
    9.449767371846567e-01     4.396373545230938e-01
    5.027334856329745e-01     9.947531032297192e-01
    5.406959105270348e-02     7.622530645858541e-01
    4.632828826489732e-01     5.553567516084107e-01
    6.016280418815296e-01     7.323617042170153e-01
    4.169700447021398e-01     3.889612909320347e-01
    3.011455519215732e-01     1.353231892378289e-01
    8.771453637021158e-01     5.950228865479326e-01
    4.458412324118016e-01     6.186721281318379e-01
    3.397389669159196e-01     8.537627574934775e-01
    3.244744579154375e-01     5.594738535316834e-01
    7.170048844809304e-01     5.257736963675258e-01
    1.503090326492018e-01     7.512144167807587e-01
    8.683580452286971e-01     4.878453537795276e-01
    3.015011387874913e-01     1.102569051974672e-01
    9.524384477147682e-01     5.015972022550580e-01
    7.780338669940721e-01     9.623247172437521e-02
    3.194160514684582e-01     2.374479297770899e-01
    5.558811653981407e-01     5.434352909392794e-01
    1.672861374817599e-01     7.835298443651636e-01
    3.514453886230067e-01     6.637236064188634e-01
    8.488487208994826e-01     5.053430777109388e-01
    8.527934218328038e-01     4.748578584553095e-01
    5.429821131500557e-01     3.730170209475095e-01
    5.211485632776544e-01     4.327662153720940e-01
    3.300086535112363e-01     1.763152481477033e-01
    4.546094353333651e-01     9.457125006720597e-01
    4.182055116126111e-01     8.820488013451818e-01
    5.668666152509535e-01     9.721158866583771e-01
    4.335329851425800e-01     2.877052478702662e-01
    7.114501849902781e-01     7.334134403370194e-01
    6.350304325886711e-01     8.771035393668757e-01
    3.560780568968139e-01     4.637905835970050e-01
    9.324402354680278e-01     5.132801812030913e-01
    4.289214129746335e-01     6.853753226533752e-01
    4.087421880736802e-01     6.088005466678812e-01
    5.662988898046269e-01     4.968173505952275e-01
    5.060625596335764e-01     3.989990444914099e-01
    8.768635714174134e-01     5.119295437750673e-01
    4.358842154850378e-01     2.661871523274351e-01
    2.256858030482731e-01     7.390264718255430e-01
    5.381526157208626e-01     5.443776546899696e-01
    6.377644696944362e-01     6.922808704360982e-01
    4.089404247317402e-01     1.677167558984551e-01
    5.642286154870292e-01     6.139904186612196e-01
    5.303877421522103e-01     3.452150379419298e-01
    2.322656411593924e-01     7.565105587930981e-01
    8.009183482046504e-01     6.755306316869483e-01
    6.266652172191544e-01     4.902672431778378e-01
    3.532518927434776e-01     2.811617920328452e-01
    6.479125297654852e-01     4.504773921929641e-01
    6.299817593831619e-01     4.202687564893330e-01
    5.393229735277281e-01     4.032897215907296e-01
    3.464907984446121e-01     7.415564354628427e-01
    3.997153344558253e-01     3.763398862612335e-01
    4.859304042932558e-01     2.387271663315286e-01
    6.196928284346964e-01     8.993394135307911e-01
    8.838339016752850e-02     7.686921321889637e-01
    4.605081670259406e-01     4.486868592949333e-01
    4.727912080074758e-01     2.864249494910808e-01
    3.829176156434296e-01     6.348766981230836e-01
    6.844861483388863e-01     6.859139586672087e-01
    2.670861599082195e-01     4.782799135125018e-01
    5.688292113669028e-01     9.137621194185365e-01
    4.229724453731069e-01     3.268714467790474e-01
    7.787598371518225e-01     5.843830358104709e-01
    7.911743233322697e-01     3.742674720899102e-01
    3.075388325414713e-01     5.270032062158109e-01
    7.749201333830066e-01     2.827704072346306e-01
    3.090827287414706e-01     7.372768671711249e-01
    4.249447941727710e-01     3.729608250795619e-01
    9.586810780705179e-01     5.139723139184532e-01
    2.542916792987513e-01     3.203980357997742e-01
    5.372773854945047e-01     9.322991674549291e-01
    2.247424529063488e-01     5.979307163944577e-01
    8.763686542771740e-01     3.497186781963703e-01
    4.908842401741225e-01     7.764319322065438e-01
    5.387891843089782e-01     4.034431347694206e-01
    3.175984220513821e-01     1.994418753219798e-01
    7.823606050170849e-01     6.929545670075328e-01
    2.155922011416648e-01     8.094574755786811e-01
    6.581729634361480e-01     8.662670492879010e-01
    2.992285272302991e-01     2.492870355880021e-01
    1.625176180201630e-01     8.051310313703417e-01
    7.848285267760308e-01     4.569552690511944e-01
    4.487889649202060e-01     2.298164669909606e-01
    7.270954068997920e-01     7.271142190589426e-01
    5.246298466355507e-01     6.752613083185446e-01
    5.402451806033168e-01     5.666412641737070e-01
    3.002613334832862e-01     2.482358496084607e-01
    5.482379745383942e-01     6.777991470505611e-01
    8.344882373083818e-01     6.012004898135751e-01
    4.135677806004827e-01     8.531699761438537e-01
    6.467325103236086e-01     8.406353050345708e-01
    7.362927167517382e-01     5.882249435412009e-01
    3.452171253286077e-01     8.748075543164454e-01
    2.655981341539944e-01     9.279522329392975e-02
    7.738337279785158e-01     4.971388346654925e-01
    6.327579365940457e-01     6.764973817478519e-01
    5.574743718580623e-01     4.567462241589327e-01
    7.826181961829933e-01     5.573952060839167e-01
    1.513766543546838e-01     7.136506793493989e-01
    4.156705980897784e-01     3.894216483550608e-01
    5.420998279559252e-01     6.964571076406678e-01
    4.377403282194727e-01     8.420412350965789e-01
    7.368729007278539e-01     8.355666832874681e-01
    8.540939160152885e-01     5.552415406549969e-01
    7.207722093312265e-01     8.483253494048318e-01
    4.459133570670644e-01     7.653145158848753e-01
    6.405876817716559e-01     4.698270159879158e-01
    1.957432138058243e-01     6.917441993735938e-01
    3.460073699975887e-01     4.310922657863027e-01
    7.377020932767338e-01     9.913715387991143e-01
    9.770703199605379e-01     4.849404591169265e-01
    4.291990785052930e-01     6.723769646087988e-01
    1.791395726480335e-01     7.698760683770807e-01
    5.438491500754931e-01     9.739568480127707e-01
    4.703384966939254e-01     5.354511113067233e-01
    5.056295156699495e-01     7.412625980065135e-01
    7.990937715537210e-01     4.948300951005413e-01
    3.255423266222450e-01     7.595124960799183e-01
    4.374695428826955e-01     9.572186611764729e-01
    5.348546433780391e-01     4.916780599565687e-01
    8.492826843482548e-02     7.793991671956807e-01
    5.425298956515371e-01     6.676584255658113e-01
    6.640946911674422e-01     6.292865742157528e-01
    6.731163549259951e-01     8.544602502430229e-01
    5.552666210909042e-01     9.824663900336925e-01
    1.933603754639495e-01     8.308950052312540e-01
    3.860416494846560e-01     4.553645043478840e-01
    2.196149559169049e-01     6.060412250554192e-01
    7.231763403469897e-01     6.323132231691759e-01
    2.820030184928716e-01     3.673211090869004e-01
    2.513457665361353e-01     6.931738999042525e-01
    8.316636019901303e-01     1.635016560656816e-01
    5.992347757717511e-01     7.589268046053712e-01
    7.082960510325876e-01     6.311776102604282e-01
    4.567361197216019e-01     2.558695202191396e-01
    2.642377303394945e-01     8.007112556342363e-01
    2.960568124807984e-01     7.832491251059343e-01
    2.831761713128093e-01     9.977773400631273e-02
    4.225792761111284e-01     3.700942996712623e-01
    2.838794428372077e-01     2.553537832386943e-01
    4.121805528707586e-01     8.771261702741593e-01
    4.231462299049473e-01     5.872805947962150e-01
    2.913680895163930e-01     5.511410731967200e-01
    6.326593098111575e-01     8.439419272610095e-01
    4.511837959389298e-01     8.736538660698910e-01
    8.405231730300543e-01     2.041600853739985e-01
    1.799886602181213e-01     7.113120549122727e-01
    3.107567099894852e-01     1.139378967381034e-01
    5.878578071112790e-01     6.391658427679069e-01
    8.593780623571431e-01     3.156417978693117e-01
    5.814513170641945e-01     8.812175095695667e-01
    3.935290478107281e-01     4.237071390149054e-01
    4.120544571214991e-01     9.452340891729490e-01
    9.255875807601985e-01     3.962464264381090e-01
    3.167095212722808e-01     5.608612713905449e-01
    5.260482551736616e-01     4.331811827478883e-01
    5.506056725829125e-01     6.858788603456598e-01
    7.420745559416427e-01     5.567942845415523e-01
    8.404837269592164e-01     3.396474516273112e-01
    5.348777212530146e-01     9.160855506603991e-01
    4.648876875893531e-01     7.434983587925368e-01
    4.989840838152515e-02     7.396075286194334e-01
    7.719947686982267e-01     1.382389070448613e-01
    7.600697841617313e-01     2.332667861876643e-01
    6.234254187506542e-01     6.923433358162224e-01
    6.921824338460658e-01     8.620656593515993e-01
    5.552165053576575e-01     8.460661379385818e-01
    5.383771926854684e-01     9.119546607304748e-01
    3.247799080534561e-01     6.808906230051349e-01
    8.494972714082684e-01     6.454917991710127e-01
    6.317534113066915e-01     7.204212164786130e-01
    6.634757984778143e-01     4.861692875744273e-01
    9.937494639789548e-01     4.911191496483998e-01
    6.870707728385712e-01     8.474946696755083e-01
    4.892509046857856e-01     4.520049376787530e-01
    7.714151201491286e-01     2.111948151330330e-01
    6.373906421422804e-01     6.075370973534329e-01
    6.818670081116077e-01     8.567318364289955e-01
    5.625705173704320e-01     6.283070571884792e-01
    5.333480690330431e-01     3.912469641703205e-01
    7.481198920117259e-01     6.818887098352919e-01
    8.017341098505848e-01     3.391043486285397e-01
    9.601262932194011e-01     5.044136551258053e-01
    4.441691303207491e-01     4.764618600270709e-01
    8.692146207661182e-01     2.935909914683249e-01
    3.856611996465847e-01     6.771404761697905e-01
    7.479355200733139e-01     6.470214449775367e-01
    6.887544886065013e-01     8.737545925441275e-01
    8.757344503813554e-01     6.081911856521995e-01
    3.566816325258665e-01     5.828066854905095e-01
    2.483806848041087e-01     7.546907016704011e-01
    4.772751082920528e-01     8.753028995457277e-01
    3.248209996703447e-01     5.483179331670385e-01
    8.098864535697764e-01     6.880428616269225e-01
    2.195692815776483e-01     8.433393786536211e-01
    7.069715547909422e-01     9.724969917066478e-01
    7.610846628575491e-01     1.194310418373582e-01
    1.761723865772498e-01     8.109686377355124e-01
    5.284624974973742e-01     7.462432955044289e-01
    1.993832821412629e-01     7.122995204694056e-01
    3.529715840186457e-01     4.812295481072055e-01
    4.306177038888512e-01     9.027737754396440e-01
    3.305227360574086e-01     4.001782261584556e-01
    7.721913390172761e-01     2.341242078561218e-01
    4.286320726883339e-01     2.488046323810311e-01
    3.169527009819021e-01     1.033970766004427e-01
    4.810090107800421e-01     8.023613946701093e-01
    8.039591640557791e-01     2.445958363735881e-01
    3.691613163764997e-01     1.964767000502595e-01
    2.995197439477290e-01     1.195273876578548e-01
    6.403324913157460e-01     6.959530188896698e-01
    6.395876253910029e-01     9.290848019530077e-01
    1.053911952836336e-01     7.630174679573457e-01
    8.038943401504451e-01     6.294646204795404e-01
    4.600283749902627e-01     3.056408353904426e-01
    5.047426722100016e-01     2.821090981101169e-01
    7.770175214105851e-01     4.981400111290227e-01
    3.721974284582960e-01     1.291265670995640e-01
    5.122755504633499e-01     4.262102217575644e-01
    4.529720833521692e-01     5.830151954539067e-01
    4.009055395860349e-01     8.323979702613931e-01
    2.525364280613591e-01     2.020602402437764e-01
    3.817675079969870e-01     9.375232494782522e-01
    3.616827205314039e-01     7.700852548821118e-01
    7.564538106476681e-01     3.490044270656971e-01
    3.971455650417042e-01     7.411077045402424e-01
    8.177014007981969e-01     2.448941953636392e-01
    2.223521837575343e-01     5.543966004111682e-01
    8.601911193928369e-02     7.086894021705585e-01
    7.544850483671721e-01     4.116928866764271e-01
    8.218599290742674e-01     2.721354648583011e-01
    5.089289481029962e-01     3.850041983105913e-01
    4.695502674506525e-01     5.530965250781688e-01
    3.781538971059695e-01     3.755100535227467e-01
    8.269894215907223e-01     4.740097706979375e-01
    5.348977062172291e-01     5.558618950726575e-01
    8.258123519584942e-01     5.868665522607487e-01
    4.871195351388373e-01     6.374565984582067e-01
    3.250104758131088e-01     3.197512721042506e-01
    1.516540938519837e-01     8.100001485478449e-01
    3.679414279204358e-01     5.476935714121973e-01
    5.830725621407531e-01     6.292743909785334e-01
    8.517428475824707e-01     2.125200972860500e-01
    6.720825489613513e-01     7.855538836794949e-01
    6.410221667653162e-01     6.471260320426695e-01
    2.575952328902746e-01     4.151719304306212e-01
    4.501105220987106e-01     8.056405928614403e-01
    3.477973506813220e-01     3.595910948464637e-01
    8.222108296628060e-01     4.641387575860315e-01
    9.693101840116417e-01     4.520977394807079e-01
    2.552416656880110e-01     4.831457210560530e-02
    1.070747105306817e-01     7.936861080075136e-01
    3.750547532854916e-01     1.805236411966883e-01
    6.652503425976051e-01     7.329115478724728e-01
    3.151309928821568e-01     2.875137797821260e-01
    7.523990618426993e-01     5.111192752749301e-01
    7.958350969477340e-01     5.997684387983855e-01
    7.800279721228310e-01     5.319592814554084e-01
    5.423267676302834e-01     7.396509233250488e-01
    4.861841021188582e-01     9.165253293497686e-01
    2.131476312441265e-01     6.876414093698191e-01
    3.859931629176203e-01     6.491801028815980e-01
    5.223499260945939e-01     9.261128472430835e-01
    4.386817225912458e-01     7.911575308645654e-01
    7.661857536312282e-01     2.808609282852320e-01
    6.432561820776279e-01     5.981689229238737e-01
    2.263234820373310e-01     6.469868237859906e-01
    4.822974729152669e-01     8.596925919393383e-01
    4.044789648399274e-01     7.921809654326959e-01
    2.592027756664448e-01     1.946169321875395e-01
    8.185544011070067e-01     3.586597344237104e-01
    3.536547464998800e-01     8.201686461161783e-01
    4.214502243979064e-01     8.209590974117897e-01
    4.517477610079936e-01     7.164445593376492e-01
    4.327938052397389e-01     7.405237609461535e-01
    3.558522001580057e-01     1.171775205793310e-01
    1.021470121431307e-01     6.487085674871640e-01
    2.364796144361345e-01     7.496264853158496e-01
    2.871152982443815e-01     5.653507864130536e-01
    6.688477400949834e-01     9.510792781254281e-01
    2.491652112005494e-01     8.449312823103980e-01
    5.707898809792538e-01     7.745547690848746e-01
    6.541128160323141e-01     8.207146850613809e-01
    3.075425505837464e-01     6.508891760128130e-01
    8.411512952634594e-01     5.898710573860717e-01
    2.700011998404225e-01     6.017071797942211e-01
    8.332355824268723e-01     5.150480773108422e-01
    2.658682306127964e-01     6.351875887872483e-01
    7.155286346455481e-01     6.475789614853396e-01
    2.866659154311169e-01     8.429211383212961e-01
    6.562605252569822e-01     4.437191941925624e-01
    1.626549915505787e-01     6.266795057011364e-01
    3.180825289185895e-01     7.371814062787068e-01
    7.551516724498325e-01     9.357313737167150e-02
    4.810453760829277e-01     7.312530823953297e-01
    8.100162227613170e-01     2.043024546954237e-01
    5.250049486736943e-01     7.015909919717856e-01
    3.303256681058669e-01     2.516031301572994e-01
    3.903253091607186e-01     7.937330978880368e-01
    3.654556875284198e-01     3.730349994265481e-01
    6.032364765954661e-01     5.012586874862195e-01
    3.103934125314284e-01     6.710204475513936e-01
    3.761684564307766e-01     6.787370673896859e-01
    4.455926651191087e-01     4.523763826920800e-01
    3.732417701267897e-01     8.046480612304262e-01
    5.204948928268157e-01     7.969379719817248e-01
    4.836071040522565e-01     3.449805949860607e-01
    1.057856472950831e-01     6.614272722581067e-01
    4.440430896472555e-01     8.951452636548123e-01
    4.739021248282042e-01     2.885128731693615e-01
    4.391656712520723e-01     5.561924361780608e-01
    4.335132913803521e-01     9.527322491940720e-01
    7.174115966223793e-02     7.418016255920750e-01
    8.046601927654611e-01     6.119716665646987e-01
    3.692938415592270e-01     3.680519259134385e-01
    8.187859013652995e-01     6.078849614283585e-01
    7.016200469097573e-01     5.075054888862881e-01
    4.753823708441253e-01     9.245985800357045e-01
    9.391276417803420e-01     4.666979985460321e-01
    3.591177329469531e-01     5.241881455642868e-01
    7.401100331502414e-01     8.640466487910614e-01
    3.775193075969909e-01     6.675269066990757e-01
    2.115150476215552e-01     7.743706995983816e-01
    2.966912301517070e-01     5.961578091782646e-01
    5.664861624152481e-01     6.117572799565261e-01
    6.653399141104769e-01     4.808068279253134e-01
    4.472972061392074e-01     7.468139761952852e-01
    5.383752884041696e-01     4.249151557419248e-01
    2.600556598748737e-01     1.312793658098548e-01
    3.727381381427158e-01     3.746954444098818e-01
    1.646329262674351e-01     6.670698729609177e-01
    6.783481709696284e-01     7.335066318515195e-01
    6.423333094533340e-01     9.400694134873746e-01
    4.109481221334239e-01     6.612265085017184e-01
    6.932293044826846e-01     9.344787182951487e-01
    6.013319053633920e-01     7.213964279652191e-01
    9.341774620040861e-01     4.282285933266948e-01
    2.034479345429473e-01     7.360135390808861e-01
    3.575159000113287e-01     3.524373388813548e-01
    2.736462263860361e-01     5.220089343201415e-01
    3.600585407743504e-01     5.999742296597079e-01
    7.359562831619044e-01     9.275804872676879e-01
    9.853954523858313e-01     5.036765215449032e-01
    5.518895883603451e-01     7.295744210209641e-01
    2.936267192485814e-01     1.329781566142703e-01
    5.680707083692129e-01     6.023469261513457e-01
    8.220657148164021e-01     1.814914110972740e-01
    1.716865022968507e-01     6.392961344609449e-01
    5.693404547034369e-01     4.078201844285178e-01
    4.437229236221971e-01     7.720005300043545e-01
    4.785635611430107e-01     5.819192284696066e-01
    7.190879523701775e-01     9.546365767522074e-01
    2.732984623136910e-01     4.455262663073177e-01
    3.224511389521708e-01     5.667790969388663e-01
    3.872856490759864e-01     3.957862879577647e-01
    7.391858389863203e-01     6.667892503769624e-01
    3.619213557699072e-01     1.852689052968683e-01
    3.306168462216961e-01     2.973860852875558e-01
    1.218096036777687e-01     6.658948546576345e-01
    3.948112144151406e-01     9.464168257971317e-01
    7.972441015830533e-01     1.576040008424507e-01
    5.654734139536635e-01     6.403181814561344e-01
    8.202119959191627e-01     6.021345206247294e-01
    5.029274570947484e-02     7.299418337276520e-01
    8.281987794368184e-02     6.898275692433564e-01
    7.123636398760392e-01     7.329926386020114e-01
    5.567178329427339e-01     9.109142494227108e-01
    4.095902061649301e-01     4.373635682470332e-01
    3.403977987251594e-01     7.951464516266189e-01
    5.451230014679042e-01     3.981801644487746e-01
    7.941652079613624e-01     1.904059479243717e-01
    4.430723631065873e-01     4.048061219440571e-01
    8.819486933417879e-01     4.526701220068805e-01
    7.136157505007668e-01     8.513386924711102e-01
    5.948459143820692e-01     6.461073464985346e-01
    5.416795312079496e-01     5.269499150856110e-01
    7.763408298809825e-01     1.507501651192201e-01
    4.841523951495227e-01     8.231150992232878e-01
    3.186864579144506e-01     3.657716894054016e-01
    1.779669315782157e-01     6.942978575327121e-01
    9.575936783207117e-02     7.271311948581195e-01
    3.036019991367808e-01     5.885430792638743e-01
    7.941523874848443e-01     1.534170981321299e-01
    8.576754831712702e-01     2.840297619624569e-01
    6.030629840378321e-01     7.446749518414880e-01
    2.697895177883670e-01     6.061531281823599e-01
    6.966094194728282e-01     9.100333163018729e-01
    3.571098782792326e-01     5.685842771195265e-01
    6.637791822655533e-01     9.152578534345853e-01
    1.490352078438428e-01     6.030074260479629e-01
    5.738163319759864e-01     8.517314820426848e-01
    5.888130140376564e-01     7.475038997945956e-01
    2.244784724858161e-01     5.728857459728564e-01
    2.443431260285128e-01     5.446122710066092e-01
    2.370319277982768e-01     6.161467958091932e-01
    4.859208179173202e-01     7.331308394864163e-01
    2.905401660118766e-01     4.798793863317456e-01
    3.557781775871396e-01     4.063040000855492e-01
    7.542755019726497e-01     4.076219617361108e-01
    6.024064069723666e-01     3.529287116519806e-01
    1.493571299989990e-01     7.331083656216512e-01
    2.842762829186323e-01     2.644224292282947e-01
    2.835975554107096e-01     4.785257472160439e-01
    2.803349691685880e-01     4.968142714000723e-01
    5.899046562408183e-01     5.646882791201074e-01
    7.037252958275250e-01     7.342898751124676e-01
    5.415759116235620e-01     5.506896671139372e-01
    6.026069192138216e-01     4.065903372970370e-01
    5.220623274410567e-01     3.835574100914505e-01
    2.412390417572274e-02     7.361182861794034e-01
    1.975046785897887e-01     7.985635486877887e-01
    1.901169639731316e-01     5.625216995689681e-01
    4.494438049520527e-01     6.642231764204120e-01
    8.044008812484499e-01     5.665593345037088e-01
    8.283806378508711e-01     3.253423727067196e-01
    8.065682318586912e-01     4.629038241369169e-01
    4.097280229259724e-01     6.507605270878123e-01
    6.458105886650737e-01     8.471908538887811e-01
    7.364042690951040e-01     6.334480025849307e-01
    3.030894584025615e-01     7.183275248935026e-01
    5.885002941339086e-01     4.114286002249372e-01
    2.997799910728962e-01     1.117229165217197e-01
    2.464009664913789e-01     6.922090921321815e-01
    5.750881829577708e-01     7.473280824122230e-01
    5.862017974775910e-01     7.473093570744473e-01
    2.611397998655631e-01     7.648660901745526e-01
    4.783803382910188e-01     7.593687362065641e-01
    2.745520408141574e-01     5.438664157803617e-01
    4.213183035993270e-01     4.981668288322008e-01
    7.128501687231802e-01     7.901389531840221e-01
    3.592374707395327e-01     7.309690574330984e-01
    4.096790635851754e-01     8.343083648640330e-01
    1.115441533885152e-01     6.719345219068070e-01
    4.627546786525785e-01     8.627769350420905e-01
    2.087224548996077e-01     6.414627425670927e-01
    3.606155409886074e-01     8.230867678948900e-01
    3.412574989439585e-01     2.659164752938358e-01
    2.759534640095533e-01     6.783672226455864e-01
    6.694910262789573e-01     5.675530787962869e-01
    5.523302007892457e-01     3.327139304395006e-01
    2.538551400004194e-01     3.233117340988964e-01
    5.620984953863123e-01     9.864549244013123e-01
    2.782701662070728e-01     7.026389647770757e-01
    4.603384413554843e-01     4.767673934056845e-01
    5.358782548409760e-01     5.843089173630154e-01
    9.337423044717093e-01     5.400265299122949e-01
    3.370618953136618e-01     8.858658899939598e-02
    5.205696692793087e-01     8.791518017345032e-01
    8.092903211465764e-01     4.655817721160751e-01
    3.808702673243343e-01     7.553404279124118e-01
    5.268238889462677e-01     4.565962339057580e-01
    1.470019873836201e-01     7.306145554085230e-01
    8.019301565785222e-01     2.264624657013884e-01
    4.156343431740427e-01     4.813260348628131e-01
    2.679265468167650e-01     8.128641898184567e-01
    2.836601771855558e-01     8.710877059226514e-01
    6.055243838600558e-01     8.199827878399091e-01
    2.621062293653507e-01     7.508193129760833e-01
    8.017623525125419e-01     4.912787838672931e-01
    8.231421317318544e-01     3.068543279512853e-01
    3.839293845272129e-01     4.125864813361351e-01
    4.416355344001625e-01     7.968178891532502e-01
    1.935519311337022e-01     6.791484535567258e-01
    4.503270256056099e-01     7.665963563309049e-01
    7.167373343010615e-01     7.123733093760307e-01
    2.708320145821573e-01     7.086844224925212e-01
    2.733835804147248e-01     6.684608517276082e-01
    7.582053685567128e-01     9.293837971314334e-02
    2.306341916311194e-01     8.425857298315979e-01
    3.466947289464305e-01     4.199954019006340e-01
    8.429497859813772e-01     6.141506626514459e-01
    4.057323423076894e-01     2.951276327042716e-01
    2.677016486735222e-01     4.321871201845424e-01
    3.810516886044871e-01     5.247093546780136e-01
    8.260480967229832e-01     4.296383731112188e-01
    6.151391802406203e-02     6.894813693070092e-01
    5.663409443290368e-01     7.459576081461944e-01
    8.232664378616450e-02     7.808676472721082e-01
    4.003046085105966e-01     2.069007226433417e-01
    5.427474647044468e-01     8.004021945583703e-01
    4.232113918333464e-01     9.574340480612625e-01
    8.911105013693911e-01     4.567408283984713e-01
    3.508846639774406e-01     1.192867719549264e-01
    7.107386873769181e-01     6.008407712135784e-01
    3.520247002983784e-01     3.690925464939834e-01
    4.286078948840570e-01     6.499276767372619e-01
    4.829653283562992e-01     3.244622248880533e-01
    5.351807711244465e-01     3.100726989526621e-01
    3.094871490948562e-01     2.359274937216894e-01
    5.685292072455105e-01     7.795255698675929e-01
    7.344814923712636e-01     7.558024251645966e-01
    2.590617481656691e-01     6.575551172765510e-01
    4.070447556157853e-01     8.085374669379788e-01
    7.957841946073655e-01     6.190013144594074e-01
    7.767991425056248e-01     6.080603670421573e-01
    7.933500927266826e-01     3.053650338204378e-01
    7.014262630496904e-01     7.509880490263732e-01
    7.013554085233232e-01     5.171904696340681e-01
    5.121183547497993e-01     7.517084190645954e-01
    5.512696544247174e-01     9.085731987760091e-01
    4.264262264244679e-01     5.513644700957372e-01
    4.031077107006144e-01     6.753915334385685e-01
    5.120550703184950e-01     4.275988386946749e-01
    7.586116723530820e-01     4.746867416247476e-01
    5.478239656857747e-01     9.829159051899604e-01
    7.611795383547777e-01     3.390103483872087e-02
    6.652554101748975e-01     9.759571801992283e-01
    5.867738093824609e-01     8.572620752665114e-01
    3.091426839832779e-01     5.687508055374211e-01
    2.258313195643578e-01     6.889916221891199e-01
    8.189366777152104e-01     6.423982777442471e-01
    9.431212089397347e-01     5.214660945941004e-01
    6.130301916237865e-01     8.314972884366578e-01
    4.804061565860387e-01     7.903206643528077e-01
    2.434651299437247e-01     5.862466558298217e-01
    5.044004111353620e-01     3.245375237192217e-01
    2.271194353789057e-01     7.902920816895703e-01
    8.598575086044116e-01     6.246687784892074e-01
    7.367854102674307e-01     8.771813261838830e-01
    8.211331085529018e-01     4.115494539452960e-01
    7.989566873318681e-01     4.716258062177561e-01
    2.617219935450559e-01     4.597142588350317e-01
    6.178610361144542e-01     6.345261437655425e-01
    7.769162480723700e-01     6.209949536850009e-01
    5.806960094302754e-01     4.331633948889194e-01
    8.312969746788772e-01     5.561338903591307e-01
    8.381106290390822e-01     2.936666329851798e-01
    6.845726938451612e-01     5.604405139676224e-01
    3.329188351783264e-01     2.722353821964494e-01
    5.745385905017772e-01     8.163483268768477e-01
    3.855133766195608e-01     5.133783438721725e-01
    3.381322224517600e-01     6.021624070638877e-01
    6.624110731418862e-01     4.590895090025136e-01
    3.260096261143416e-01     4.131028014099204e-01
    2.478800839279175e-01     7.502422592524457e-01
    4.145357855797193e-01     3.618025271252207e-01
    6.458735184461210e-01     4.954626306286443e-01
    3.463550213720636e-01     8.125259592654626e-01
    5.913813393203116e-01     5.085373565427281e-01
    3.466238369475969e-01     1.636355810227093e-01
    8.082687824882234e-01     2.498607865093122e-01
    2.145138684149884e-01     8.043205446466148e-01
    5.516314765999623e-01     7.568241580854802e-01
    3.648877919689923e-01     4.297910570249766e-01
    3.143717354994850e-01     7.933281778100205e-01
    4.125890665847037e-01     9.127339712755370e-01
    4.812352955614025e-01     4.371341781150538e-01
    2.758672524003282e-01     3.716309069478536e-01
    6.087314699675161e-01     6.884211269915606e-01
    5.664344750638575e-01     4.858500159587971e-01
    8.581733288984057e-01     5.511285164110169e-01
    4.970946769552680e-01     5.906852933360343e-01
    5.174355903185177e-01     8.117928020679708e-01
    1.703255862571953e-01     7.099095348130675e-01
    6.688822169387628e-01     6.271305401696566e-01
    5.899973836768992e-01     7.209833178373516e-01
    7.221857324049588e-01     5.602634381078598e-01
    7.276300366149553e-01     7.264284916184377e-01
    7.995546755375151e-01     2.133202772348807e-01
    3.067388514871736e-01     5.222225037443284e-01
    4.718138044728711e-01     7.145351618573519e-01
    7.583678959217121e-01     4.792312504256340e-01
    2.629616138313026e-01     6.648649247246358e-01
    3.122592418186735e-01     3.605343321434300e-01
    3.183527675324027e-01     7.759898906285996e-01
    2.685588152164889e-01     8.291172759635020e-01
    2.879915793347060e-01     7.906065169495559e-01
    7.384133966963338e-01     8.091561790169824e-01
    7.090457391293077e-01     8.332714417216808e-01
    5.283184022788415e-01     4.306972291071556e-01
    1.721418872565573e-01     7.408978945806013e-01
    7.466778955927393e-01     9.115456827698080e-01
    8.520700391397513e-01     3.528204024486391e-01
    6.870365448557983e-01     9.294577815826955e-01
    8.065772888443082e-01     5.219438660578608e-01
    5.778982985080992e-01     8.775778510496254e-01
    4.620418878050576e-01     5.784925565435513e-01
    8.467622340636456e-01     5.530583924063520e-01
    6.145644704285802e-01     9.991357523904499e-01
    5.553683280357332e-01     9.690906273968427e-01
    7.945097703599165e-01     2.320912753682605e-01
    3.660133883700071e-01     7.547184086369439e-01
    8.100767149402520e-01     1.987823272557822e-01
    2.508827697205918e-01     6.300565486550034e-01
    3.634216452956300e-01     5.558097862802187e-01
    5.040081579627339e-01     9.272749191419322e-01
    5.496185424469197e-01     8.764340536385371e-01
    8.304755193673002e-01     2.364650555117908e-01
    8.440831814124363e-01     6.475056218697620e-01
    2.995742914987679e-01     4.362911698058768e-01
    5.102696207332607e-01     3.437000054328271e-01
    3.514972839107018e-01     5.343700295119652e-01
    7.856534454038464e-01     5.304919145848078e-01
    5.283082111144237e-01     8.310300261840378e-01
    5.599835843139520e-01     3.780208787338982e-01
    2.630829921321375e-01     7.015577649785405e-01
    2.511943101042571e-01     5.290898260845290e-01
    3.168601819594065e-01     5.577745820950742e-01
    8.164049958591786e-01     3.240294063029028e-01
    3.005788176573884e-01     1.416915914730766e-01
    2.548923816504918e-01     1.747424830484366e-01
    4.396908894539470e-01     3.574446660914558e-01
    4.151709082264515e-01     7.059656604266944e-01
    9.742365733377248e-01     4.910993856966623e-01
    5.584819118080566e-01     7.076258853516082e-01
    2.355972183138545e-01     7.059484537653781e-01
    4.459755509207357e-01     4.523156091452950e-01
    4.439454108965352e-01     7.439713638739467e-01
    2.915878094108909e-01     1.044258657172538e-01
    2.775913345045454e-01     3.440907145023784e-01
    9.211307467244227e-01     3.778116989439095e-01
    3.491813573420610e-02     7.442656841155258e-01
    7.857880599810257e-01     6.010211024405152e-01
    6.622885956027813e-02     6.986628681470262e-01
    6.716998026736466e-01     8.301221639620510e-01
    2.077650085192734e-01     5.880643057873443e-01
    5.487519653015581e-01     6.638537103134557e-01
    9.101545751580332e-01     5.536467176342970e-01
    1.858203576300729e-01     7.870410900402671e-01
    8.140128491282128e-01     2.346358067260236e-01
    1.603946137944776e-01     7.552055394569986e-01
    9.339388046376386e-01     4.644857479215415e-01
    1.931098895342355e-01     6.900336726155816e-01
    5.993153493891078e-01     8.215129034627447e-01
    4.206435649741247e-01     7.637085826661194e-01
    7.340816121379884e-01     6.762795979776730e-01
    2.683343170952308e-01     6.839519092680614e-01
    5.325237197551210e-01     5.569278367984023e-01
    5.286975786303297e-01     8.419822839168469e-01
    1.823904841659307e-01     6.453227104532722e-01
    7.606367506905967e-01     5.517770657482509e-01
    5.581968119027226e-01     3.828084266885322e-01
    6.253961657544036e-01     6.617824664220388e-01
    4.748341531624062e-01     2.815050994937796e-01
    2.406413586338123e-01     7.391958198119455e-01
    4.147475058179304e-01     9.421510345024801e-01
    4.074933083749437e-01     7.824714823793326e-01
    7.080505136978948e-01     9.527421784135005e-01
    3.509397274839087e-01     1.695019882569554e-01
    3.367957202112728e-01     2.298976440274386e-01
    5.500974337138289e-01     5.586453956022263e-01
    2.066068524238510e-01     6.085475995919194e-01
    5.159254630493492e-01     5.867776624352192e-01
    6.923995317197176e-01     6.849424441990651e-01
    2.586733593648216e-01     4.806077881466733e-01
    4.718457524100574e-01     8.645391396266229e-01
    6.293801922374428e-01     7.800879059367997e-01
    5.110472521234688e-02     7.633451062428321e-01
    5.345989419658099e-01     3.888745602437160e-01
    6.085077905590504e-01     6.488232285864088e-01
    3.311494116455049e-01     8.894367403546176e-01
    7.857812198718763e-01     2.567742087686983e-01
    6.894423055611785e-01     4.850189526384213e-01
    8.321807487726071e-01     3.120841091212870e-01
    7.481278217870235e-01     9.879670894253628e-01
    8.611404898638150e-01     2.324372939901694e-01
    5.387769132378837e-01     7.422581407977230e-01
    3.777292564099873e-01     6.333311627052884e-01
    3.805836051741492e-01     6.122872390281137e-01
    3.118061233340019e-01     8.829898335622312e-01
    6.241451841065704e-01     3.805597683226993e-01
    1.337115169464966e-01     6.293207754561145e-01
    4.178895196355025e-01     5.919902458419226e-01
    5.990150761271616e-01     5.489650159542161e-01
    2.934349849970302e-01     2.550787265474999e-01
    5.871738552019830e-01     7.447545206774586e-01
    3.563664174907766e-01     7.745326830657474e-01
    8.317508843714091e-01     5.287039071208056e-01
    7.902380876633172e-01     5.672025788830302e-01
    2.065013590198004e-01     7.651428970652289e-01
    5.931844645956376e-01     6.983544651433450e-01
    7.844781551391786e-01     6.934817737606214e-02
    3.836598693059140e-01     4.393168227375424e-01
    7.477357563267301e-01     8.444027127280738e-01
    8.260046357780516e-01     5.818542798288573e-01
    7.048698548248359e-01     8.858299443958988e-01
    7.582135154154535e-01     4.229271972978864e-01
    1.713369195075740e-01     7.037979979112317e-01
    6.658611307353060e-01     5.234825640062994e-01
    5.419532697469956e-01     4.739170600158821e-01
    3.710304454374566e-01     5.529338558831208e-01
    8.139279071624136e-01     3.528327381248870e-01
    2.585801152059097e-01     4.669525656873577e-01
    4.106265480405443e-01     6.256525220559033e-01
    2.822865385004694e-01     2.977864395370531e-01
    7.491803594378269e-01     8.892414493088964e-01
    7.477959923758738e-01     5.728183802317889e-01
    3.863142510785891e-01     2.732506881464166e-01
    3.411822956127529e-01     6.137631184341312e-01
    2.483673500200891e-01     7.649015482789178e-01
    4.096548378934570e-01     2.214996980729622e-01
    3.513590709969516e-01     5.863098268522866e-01
    5.580026866695604e-01     3.714832745819130e-01
    7.801415784872625e-01     6.358863793946656e-01
    5.067916515609548e-01     2.810219031692979e-01
    8.177811499870244e-01     5.234666660874651e-01
    3.135784097773548e-01     1.121064390245953e-01
    4.688942618362155e-01     6.935135119912506e-01
    1.113761787344617e-01     6.986462260459904e-01
    5.651156485697256e-01     5.662892616053408e-01
    6.147587392854665e-01     6.669646519148565e-01
    8.246775984209211e-01     1.649054353557092e-01
    6.802704590558806e-01     4.817436500046113e-01
    2.644688063385513e-01     5.474172484290901e-01
    3.874421098150096e-01     3.997837425106597e-01
    8.790167636614803e-01     4.341913798920619e-01
    4.403653533615797e-01     2.462545624976460e-01
    2.287658742235615e-01     8.612621311359382e-01
    5.441058326235332e-01     9.754395192082234e-01
    5.077128878742928e-01     9.916339207886432e-01
    5.802672630904699e-01     4.013430546135385e-01
    3.656708304095568e-01     8.900066433959113e-01
    3.190326826185429e-01     1.345824580569568e-01
    6.606571403373915e-01     9.650992406197838e-01
    6.084282652759065e-02     7.258883691301821e-01
    3.571629010161089e-01     4.508648388021655e-01
    7.546867573639541e-01     4.293141380997301e-02
    7.076703631443715e-01     6.736033347293299e-01
    1.393780570259999e-01     6.334828821785636e-01
    3.375262398808976e-01     5.341487618199595e-01
    4.688994052479284e-01     5.035496517808313e-01
    4.289002314259229e-01     5.073596692916424e-01
    2.546434871547602e-01     2.110041303887423e-01
    2.655586773468188e-01     2.278097408647245e-01
    4.682553572187038e-01     4.242336245081880e-01
    1.738674887804617e-01     8.333636505895440e-01
    6.634308903197982e-01     9.507127832877924e-01
    2.888887528131781e-01     3.762134982491976e-01
    2.694786264835854e-01     2.602720502100848e-01
    2.349097410571408e-01     6.707761600029436e-01
    4.463324079218967e-01     5.594827223907461e-01
    3.185262874360837e-01     8.646982079116045e-01
    7.219464465100630e-01     8.007157496414298e-01
    1.576940178241311e-01     6.731874390078284e-01
    2.992565328338925e-01     1.057989981258410e-01
    7.457339018018813e-01     9.512764413034148e-01
    3.772146560032689e-01     5.218674429099044e-01
    8.897559970044909e-01     3.464717258739969e-01
    6.841978298705594e-01     7.967932717693138e-01
    4.441840295211724e-01     6.960992476321022e-01
    6.986874552361939e-01     6.767662915536143e-01
    7.014511001308654e-01     9.368411558515953e-01
    1.213685047551882e-01     7.068193534276134e-01
    7.942870586342452e-01     4.565746101590068e-01
    9.098013717074062e-01     4.064697119588846e-01
    3.010265170792101e-01     7.264351934568806e-01
    1.459679723894897e-01     7.909964519951920e-01
    8.233364083365026e-01     1.798367694357718e-01
    7.064720689853899e-01     6.328429697025479e-01
    5.357546567622279e-01     7.620558822060873e-01
    4.707001908969223e-01     3.882747012911495e-01
    3.974785944337732e-01     8.154150314522693e-01
    8.466189621082011e-01     3.452235158755818e-01
    5.896200940356644e-01     6.592997164006438e-01
    9.058760137078972e-01     5.432648922692765e-01
    3.370457077468503e-01     3.820964239278510e-01
    4.609074259428123e-01     7.588231597182711e-01
    7.314179896282158e-01     4.851788931944859e-01
    4.887271758207448e-01     6.936130718676949e-01
    4.300128073896379e-01     4.497976214975737e-01
    7.363220929233349e-01     9.179037699322602e-01
    7.165757743378486e-01     7.680234726593118e-01
    5.569784191540459e-01     5.908692570300115e-01
    9.224731418788739e-01     3.642758560811090e-01
    4.134982277464210e-01     4.324343628288808e-01
    4.453874548982896e-01     8.344950993333132e-01
    6.519813004236845e-01     9.424236954892943e-01
    3.778169087865577e-01     5.300494025637360e-01
    6.241277849192441e-01     8.470019884136932e-01
    3.684325130634958e-01     4.460505710027021e-01
    6.829227926967560e-01     5.598751346670908e-01
    2.433604031788025e-01     7.938861728212422e-01
    5.038295150186222e-01     4.060027519538871e-01
    2.849043129395992e-01     7.886483849133931e-01
    8.182914970326186e-01     1.380031320484580e-01
    4.810397270763161e-01     8.308218125994989e-01
    3.801707042760952e-01     7.722252611769805e-01
    5.056267908145512e-01     2.881079114551968e-01
    7.625558642110750e-01     6.643076900174645e-01
    2.607534791938217e-01     2.220245607910762e-01
    5.371305108533329e-01     6.629238645938957e-01
    3.932764025629685e-01     2.557005494387475e-01
    2.483218758859544e-01     5.520602659678173e-01
    3.676394675910655e-01     7.885052203487850e-01
    4.051760984342747e-01     7.938700491058230e-01
    6.848210475000488e-01     4.824705764599062e-01
    7.317672464220053e-01     6.689334854675019e-01
    2.783537491590407e-01     5.055909963679749e-01
    5.855801780410604e-01     3.646982090509795e-01
    2.444307186714342e-01     7.681047323947764e-01
    3.666558060655336e-01     6.486167662607504e-01
    6.554867315934469e-01     6.495247301478861e-01
    7.664232824516573e-01     6.411442209988415e-01
    9.232378488307302e-02     7.109686219480825e-01
    7.747356910670217e-01     4.459728649992711e-01
    3.783815454346422e-01     5.247638950919892e-01
    6.287312329493303e-01     5.689915131224923e-01
    4.378768058929683e-01     3.132017724146021e-01
    3.310783939447420e-01     3.269769725507908e-01
    8.702492411007793e-01     3.452581552840602e-01
    2.533101200771791e-01     6.464463971547251e-01
    6.795623812315349e-01     9.957992877224919e-01
    3.756980100343869e-01     2.782467303853706e-01
    5.314099034631796e-01     2.849429623878098e-01
    4.093587016906280e-01     4.424884571291686e-01
    6.180378713605822e-01     9.137827282973521e-01
    3.281705430528949e-01     2.274105167666275e-01
    7.436592709049002e-01     6.448562932718317e-01
    3.009156842979245e-01     3.645339497197697e-01
    5.291447098277513e-01     6.679945689295063e-01
    2.691385807803978e-01     6.641259101264522e-01
    6.671848228254416e-01     5.671153310007828e-01
    5.727070353819099e-01     8.585413190584015e-01
    4.635689650783356e-01     2.612443283784485e-01
    8.678154746490919e-01     5.768878598974905e-01
    5.496840289664140e-01     9.827897593473408e-01
    3.899377636682231e-01     2.182874061282459e-01
    4.339948849607925e-01     6.346049963664520e-01
    4.192079277668034e-01     3.595610641473181e-01
    7.761776328398835e-01     1.949027698744231e-01
    4.646877929707066e-01     8.773752007457098e-01
    2.880369727359740e-01     4.835163577928762e-01
    5.651535859773859e-01     8.225532999668597e-01
    5.773704534013773e-01     8.942455770177400e-01
    3.034625835728809e-01     2.190526750918304e-01
    6.229133304888098e-01     8.075259557345514e-01
    2.018511364949229e-01     8.467324715567183e-01
    3.652900598721069e-01     2.440965973920525e-01
    8.584334509303079e-01     4.007455622829759e-01
    1.861758774679495e-01     5.666612918545445e-01
    9.071079864315327e-01     4.031645218513079e-01
    8.273816057538369e-01     5.955953943195795e-01
    6.586469940633058e-01     8.084493290023788e-01
    2.654604338364913e-01     1.579148826833913e-01
    2.953534316723203e-01     3.437630790484832e-01
    6.192489442162669e-01     8.249175918442624e-01
    3.718307135025207e-01     6.558954522658267e-01
    5.704320017680153e-01     6.566322421950912e-01
    6.531301156439653e-01     4.590160818262827e-01
    2.931883904842896e-01     7.247814003646937e-01
    5.946777330005316e-01     9.794407373605312e-01
    3.629326766627013e-01     5.605713791092073e-01
    4.874095251885922e-01     6.839531724647528e-01
    7.898865785367251e-01     1.240049359091704e-01
    6.411725506987539e-01     5.186494438142809e-01
    9.067640190899648e-01     5.100287559502600e-01
    3.714805851214100e-01     5.437964464577593e-01
    4.380593863297040e-01     4.258592017998851e-01
    2.954975038848320e-01     6.581471903800524e-01
    8.454645952083425e-01     3.672429152169755e-01
    3.850724742259294e-01     7.735197340712083e-01
    4.823074853234740e-01     5.214484549799001e-01
    6.982065220775027e-01     5.198737584361458e-01
    5.986782087840375e-01     7.031472709446205e-01
    5.563656865548064e-01     3.411895821063173e-01
    5.822053907331166e-01     6.320800187562848e-01
    7.839556313424323e-01     4.960825432707671e-01
    3.468803776192557e-01     3.242071942578262e-01
    7.964761325434050e-01     2.238154933678298e-01
    4.645854451085751e-01     8.269544756492018e-01
    3.022227368771087e-01     5.924813863188827e-01
    4.187835876969572e-01     7.806821736273026e-01
    5.922134979931913e-01     5.001562654367449e-01
    7.477427390366175e-01     6.038228356205460e-01
    3.606414850893953e-01     2.410582163367854e-01
    3.888899317444912e-01     4.031428586389938e-01
    1.997539193923975e-01     6.621911025748052e-01
    2.975210452953456e-01     5.418840465197877e-01
    8.535429580634193e-01     6.342895961856718e-01
    7.494425686411156e-01     5.452001724974601e-01
    4.991703617750345e-01     8.090756987591603e-01
    3.946039370805084e-01     5.636656462530404e-01
    6.045862043178468e-01     5.885295201729208e-01
    5.402622798551279e-01     9.069764368089257e-01
    4.069334226771396e-01     5.322847525819707e-01
    2.740348310900420e-01     3.262670059673010e-01
    3.378395977520934e-01     9.677429454064046e-02
    5.623040330838145e-01     9.415182466302718e-01
    4.054669406002691e-01     5.416168388999469e-01
    2.226664203561018e-01     6.910443834557403e-01
    4.607901219986497e-01     2.466594812953473e-01
    7.792473568088931e-02     6.871387418973985e-01
    8.119949752763633e-01     4.156962009965637e-01
    3.672811251226714e-01     4.280803495142644e-01
    4.452007231494428e-01     4.430925028224999e-01
    3.928009234576230e-01     6.029534000868308e-01
    3.864178182558038e-01     5.255249492821714e-01
    7.227653534634793e-02     6.880992797375005e-01
    4.348541172835173e-01     7.352034342046478e-01
    1.847821387416565e-01     7.925666009533490e-01
    6.989651641368696e-01     6.491283284445418e-01
    5.685945856521315e-01     5.266712874065453e-01
    6.606609776970956e-01     5.377709505813603e-01
    4.467236161312776e-01     7.873285696070408e-01
    4.193026188662953e-01     8.443212009484662e-01
    6.969140090040116e-01     8.856263253024257e-01
    8.243882410342128e-01     3.380985818720083e-01
    5.126955639672379e-01     3.708627676919081e-01
    5.960865749762512e-01     5.926756399179194e-01
    3.535888955679211e-01     2.885615012978006e-01
    1.911547095282586e-01     7.806855472277328e-01
    5.397498431617214e-01     4.696905330470865e-01
    7.375180434153590e-01     9.309005116029444e-01
    9.574937786980269e-01     5.321059381636609e-01
    4.506370091994961e-01     8.344939709613338e-01
    4.892579532347905e-01     8.441003827989900e-01
    3.322078584147776e-01     2.269105494104042e-01
    8.853631734020452e-01     6.002819081725811e-01
    1.015290141155676e-01     6.948704574021751e-01
    3.942103134782700e-01     2.088033242027858e-01
    4.268208702657152e-01     7.671289615993325e-01
    2.621337040813330e-01     3.498580241983673e-01
    6.786848414728075e-01     9.656290585811802e-01
    4.787052807719258e-01     3.643883558080077e-01
    6.580681594369959e-01     8.367680540790574e-01
    6.541245947670332e-01     9.380962649349264e-01
    6.968206512972832e-01     6.024793442032017e-01
    1.426088375888171e-01     6.518153617590318e-01
    1.334332508354996e-01     7.760557676966399e-01
    7.243023827988567e-01     9.091721139215666e-01
    4.990921265831468e-01     7.911246240770312e-01
    5.285935464721569e-01     4.251947289601313e-01
    4.036626664658991e-01     7.212947128447313e-01
    3.029470250052432e-01     8.276508482282421e-01
    7.502785723778229e-01     3.525738449567603e-01
    3.017744757404437e-01     8.777111390452688e-01
    6.305622377092115e-01     5.445280891200251e-01
    5.678360847584806e-01     6.217446372069351e-01
    6.125171193774704e-01     8.781248493987674e-01
    8.138815570113508e-01     5.233989997034217e-01
    4.252264060326697e-01     2.216341056329713e-01
    5.261834330776710e-01     8.820257433869083e-01
    7.424947177544720e-01     7.979793982707045e-01
    4.590934672352392e-01     7.144000602104643e-01
    4.223500974239052e-01     4.034611078865489e-01
    6.594279658184207e-01     6.016524869321842e-01
    8.603975440010913e-01     2.451059465379958e-01
    4.616579328365336e-01     6.002436214783625e-01
    4.384860453598091e-01     7.261401459047563e-01
    5.830287752684705e-01     4.997333007232587e-01
    4.750304876223638e-01     8.029983694499759e-01
    2.461977013653928e-01     5.859781224919640e-01
    2.044535093959584e-01     6.785725080784692e-01
    8.320201421721719e-01     2.295512116800116e-01
    6.832002742065428e-01     5.929421109823052e-01
    6.220792545162328e-01     6.121983927544142e-01
    6.228022652015286e-02     7.095975167525259e-01
    3.988341563295188e-01     1.507099970948732e-01
    6.749264457320311e-01     6.528796108225285e-01
    4.113298675508738e-01     6.072531604782421e-01
    7.901103540086525e-01     4.424726099428175e-01
    7.690543760282533e-01     1.495338012869017e-01
    8.835782779065503e-01     5.434829312468711e-01
    8.449238034029902e-01     6.308540839687629e-01
    3.552138088968538e-01     4.837156072724986e-01
    3.682989134354456e-01     1.370363845624605e-01
    1.194264770787473e-01     7.449479433603298e-01
    3.682437272597315e-01     9.239736430204822e-01
    2.884043223135091e-01     5.575063636510426e-01
    5.739290204627604e-01     9.216712366038330e-01
    9.271161961820737e-01     3.787867923516103e-01
    8.634941790817007e-01     4.780980026087942e-01
    7.672273273603120e-01     7.135905581417549e-01
    3.139061964239539e-01     1.386952832216213e-01
    8.351913759751457e-01     6.310630330866567e-01
    6.940322207879011e-01     5.023162238369567e-01
    2.751794937507003e-01     3.682331310159448e-01
    2.501872344139161e-01     3.828319163962646e-03
    5.289576120010498e-01     7.012469681919916e-01
    6.944660405035811e-01     5.556963360714267e-01
    5.564347831568773e-01     4.302367713957280e-01
    4.190409207391460e-01     1.960560661620060e-01
    5.217292660001764e-01     4.428971987070437e-01
    7.418832693448407e-01     7.426859573292655e-01
    4.674961712783542e-01     8.635230482219798e-01
    8.018531592560463e-02     7.636769999965893e-01
    7.583061205013193e-01     2.863954426782955e-02
    3.482645331618195e-01     7.295045830882950e-01
    2.162985005756578e-01     5.705312361978895e-01
    4.536460719554077e-01     4.086527139082513e-01
    7.400768640744469e-01     8.363419444080699e-01
    4.947960698345505e-01     4.070696651764378e-01
    8.407340985535462e-01     2.892186792473214e-01
    3.993008343676163e-01     6.889364954106116e-01
    2.938209867560796e-01     1.861876909836984e-01
    8.002309344956723e-01     3.771220662639646e-01
    6.580838819619929e-01     5.928533000201753e-01
    6.216942561605324e-01     5.864814909703330e-01
    3.393620665745551e-01     6.958849336069368e-01
    4.663519584410325e-01     8.842938290185590e-01
    5.244697084942811e-01     9.472661502133098e-01
    2.779938183691214e-01     4.145245382179524e-01
    6.359954524432213e-01     4.776663397245581e-01
    2.511462186473150e-01     5.735407026675987e-02
    8.340140286458146e-01     2.417835896609952e-01
    4.668281431580505e-01     7.806119579658874e-01
    5.533834251523655e-01     4.325593355238616e-01
    8.644433511841521e-02     7.486600463216928e-01
    5.701352296915523e-01     7.528264751716900e-01
    4.265526827513545e-01     8.947403914741284e-01
    5.137675366359359e-01     8.282788341454626e-01
    6.857105319973265e-01     5.374721102307779e-01
    8.021762306912140e-01     6.651163623511430e-01
    3.506795484773317e-01     5.769868554785950e-01
    1.991910140371291e-01     7.299793115838293e-01
    3.242905895061461e-01     1.951073502419610e-01
    5.266980524220550e-01     3.720567085322589e-01
    2.752847787051019e-01     4.461987846062727e-01
    3.145176373772335e-01     4.247659973698067e-01
    5.370468813634871e-01     9.448845532407334e-01
    2.315644311605644e-01     5.490335584232747e-01
    8.292877909703111e-01     3.112065917763812e-01
    4.680032334938785e-01     3.161613247954957e-01
    7.757285285541076e-01     6.597708111438910e-01
    4.673320655317842e-01     7.848867122553395e-01
    3.713533288241052e-01     5.853317501875802e-01
    7.290890521515476e-01     7.353365746824551e-01
    7.807458455803677e-01     4.652538461148901e-01
    9.209863298756816e-01     4.508849201758087e-01
    4.882587818823535e-01     3.700136509386504e-01
    3.032147297646589e-01     7.961042624541941e-01
    4.208410173366719e-01     5.192985683802904e-01
    5.160140661202628e-02     7.245245338770321e-01
    4.896875084813841e-01     5.206173205151110e-01
    6.656212976191144e-01     6.681812435439165e-01
    6.365646374660359e-01     4.625434716445351e-01
    4.591056417439959e-01     9.477973091695787e-01
    7.118932287576204e-01     6.622084728698858e-01
    4.716074663614819e-01     5.361315997610409e-01
    1.267575136639115e-01     7.139103249355016e-01
    4.935293261382934e-01     7.398577865163593e-01
    5.829477704425169e-01     9.160579022207124e-01
    1.417505161992915e-01     7.483465162984121e-01
    9.193686071841910e-01     4.893028293776023e-01
    5.364190475607228e-01     5.685665320671139e-01
    5.145507956037757e-01     5.928949178627733e-01
    5.657816880256727e-01     9.847514179798051e-01
    8.448044858770505e-01     4.955766826541206e-01
    8.306350049177220e-01     5.821306698306556e-01
    3.162255564887948e-01     3.529996817170820e-01
    7.631942844508800e-01     1.746040020424605e-01
    6.268595077047286e-01     9.044367003277078e-01
    7.104433607271845e-01     5.111639384735805e-01
    6.493220645092235e-01     7.388131903143783e-01
    4.324556419911194e-01     5.674863904458801e-01
    8.595864010069261e-01     4.362727081305331e-01
    4.877260174548058e-01     9.856696677561797e-01
    2.471807631259503e-01     7.948372940704177e-01
    2.078555560703487e-01     6.640164079423481e-01
    8.242182931430357e-01     2.023630634330730e-01
    7.806325445860369e-01     6.184343823577482e-01
    3.058859874202812e-01     5.696724327807725e-01
    3.545124165407922e-01     2.736294571128143e-01
    4.250060129767648e-01     4.174649546847765e-01
    3.258365411168438e-01     6.147054402900877e-01
    6.209457029902633e-01     5.806026592222776e-01
    4.368874918377915e-01     3.626518189387008e-01
    5.024469815173462e-01     2.571903195418902e-01
    1.162924054602471e-01     7.204153876891572e-01
    7.320483043192393e-01     8.963517553690078e-01
    5.088987637867194e-01     2.593125436170014e-01
    3.911234449254697e-01     8.814419386530852e-01
    4.141897378047862e-01     1.693763729854145e-01
    8.300526842081444e-01     6.388632241446652e-01
    4.047110930713690e-01     7.997160440461062e-01
    9.358792962992711e-01     5.019903732883186e-01
    4.862528466333340e-01     9.335129926274681e-01
    4.750096341118405e-01     8.473469890738190e-01
    3.192954319932927e-01     4.724738217031461e-01
    7.200266660399314e-01     4.722699528606998e-01
    2.625577363260968e-01     1.648061324441452e-01
    5.798943637599067e-01     7.126723175794383e-01
    3.653631912183672e-01     1.743643063045751e-01
    4.498622451640154e-01     4.267158415717469e-01
    4.031359991545304e-01     6.526431221303250e-01
    3.194824593220420e-01     2.832163856756308e-01
    5.195982633922279e-01     7.087634584841486e-01
    2.161382050066299e-01     6.770407781519940e-01
    8.073341316743768e-01     5.790443970239876e-01
    5.745792001179608e-01     8.705268963420776e-01
    4.704823715555173e-01     5.553534498361355e-01
    5.074346240311538e-01     3.794096229703540e-01
    4.376445123391619e-01     1.920663720906800e-01
    6.458415403776538e-01     9.468175447061911e-01
    8.247722964541482e-01     5.579822116810264e-01
    3.972546952235765e-01     8.501754530555543e-01
    3.355327903666822e-01     4.007467575114702e-01
    6.558526697602379e-01     9.641091307752422e-01
    6.946802886395044e-01     4.623719439731852e-01
    9.095632829519952e-01     4.390168904502901e-01
    4.362617880080183e-01     4.212140362264817e-01
    6.183362278052041e-01     8.420736774242975e-01
    7.721064387433287e-01     6.251101378668080e-01
    4.464835085745286e-01     2.923179817101673e-01
    5.107743749428538e-01     5.312697939036878e-01
    6.726027792910685e-01     6.214765643557499e-01
    9.207698833289208e-01     4.404485893323163e-01
    6.781990714757854e-01     9.248205914937442e-01
    2.891938714952760e-01     1.105998954138362e-01
    3.915012309079497e-01     6.227399998280099e-01
    3.566708985764291e-01     7.325095935973757e-01
    3.951716565838287e-01     7.557549896078587e-01
    4.064595536030987e-01     1.879071540701897e-01
    3.157752975830829e-01     1.613431646410536e-01
    6.439927337894861e-01     5.341006391003688e-01
    2.131597729957593e-01     7.466001395054629e-01
    6.569787902942822e-01     9.083588767465679e-01
    7.053513944162496e-01     7.792435295283262e-01
    7.903340108391265e-01     3.547241075131908e-01
    9.221989932831087e-01     3.965322472762021e-01
    4.147014525339996e-01     7.076732172813289e-01
    8.190614902592828e-01     2.827975110157362e-01
    3.611984996233804e-01     3.731874636361234e-01
    7.052283957544599e-01     8.377034682628182e-01
    4.012900043263991e-01     1.998445551016147e-01
    1.610976804702992e-01     6.473869787119236e-01
    6.625144926773273e-01     8.469833533848826e-01
    8.476944875296482e-01     3.895720475938427e-01
    8.349110431032167e-01     4.140308462813351e-01
    7.970390789567786e-01     4.118516260256241e-01
    3.236513102016179e-01     5.053834094599785e-01
    5.219430765280579e-01     3.455289218880506e-01
    5.525852228739392e-01     6.903882289415825e-01
    3.761509049432913e-01     9.237465455067266e-01
    3.894816934455199e-01     4.003246873658075e-01
    6.528742508090570e-01     8.511891456242388e-01
    4.916454640625441e-01     3.895159013129690e-01
    2.504825656344154e-01     4.958753097781787e-02
    7.686372040376422e-01     7.202688702456106e-01
    5.159575272557172e-01     6.977406167043034e-01
    6.596432623568436e-01     7.311353212147571e-01
    6.980831150243172e-01     8.809924861699383e-01
    5.162718835289331e-01     4.207785863151409e-01
    8.419399419600975e-01     1.959840396022772e-01
    6.425064774696514e-01     5.621126307588933e-01
    3.771090276205390e-01     4.840587116364434e-01
    7.058637269734892e-01     6.836714691888521e-01
    4.195941936543731e-01     6.627828251771489e-01
    6.564557360679696e-01     6.866736737860317e-01
    5.399398957601214e-01     7.156441623368301e-01
    4.849909001205827e-01     3.431314635020397e-01
    5.148012759153566e-01     6.740988953870292e-01
    5.692460090822079e-01     9.280410781791810e-01
    3.922393177578681e-01     5.917990900538497e-01
    6.591225298618928e-01     8.927019429013319e-01
    7.967628372686754e-01     3.667313782435218e-01
    7.731509987649886e-01     4.725525646476264e-01
    8.310034485027398e-01     6.627062873538574e-01
    8.146065559368540e-01     2.732501394522195e-01
    3.374387912666625e-01     8.917543857772018e-01
    3.714547413376680e-01     5.322478686304820e-01
    2.824532455066926e-01     4.512224953250148e-01
    5.847426598901861e-01     3.856635379789475e-01
    7.792263630985381e-01     3.588499132607148e-01
    7.202050929773092e-01     7.137268868481228e-01
    5.055767795832585e-01     5.164251121186525e-01
    9.946600143755147e-02     7.531487779528498e-01
    7.762932034526803e-01     5.587617210702277e-01
    5.148383639727759e-01     5.275396060824193e-01
    4.753287567286552e-01     4.823299389348070e-01
    5.655883987131116e-01     6.168637115652026e-01
    9.257320892260625e-01     4.131551827272018e-01
    3.613329297416074e-01     2.505230073275275e-01
    8.734401538331376e-01     5.226817030125477e-01
    1.706664674450700e-01     7.176601295228782e-01
    2.705337054588324e-01     8.498233633776038e-01
    4.693575425396450e-01     4.268180264203146e-01
    2.199100527127620e-01     6.938075685734294e-01
    7.424056299694151e-01     9.696570301360369e-01
    5.832835571318656e-01     9.083521699540914e-01
    7.476197354460963e-01     8.475280092862940e-01
    2.812942121582315e-01     8.633102838118094e-01
    4.827507067042043e-01     3.820407845413715e-01
    7.426160984986011e-01     8.826112218567287e-01
    5.645249187466598e-01     7.899389188017141e-01
    1.409968489176618e-01     6.801940381976028e-01
    6.912668897156896e-01     6.870120896846661e-01
    5.991425128578846e-01     6.067541788274872e-01
    2.884721213503629e-01     8.128084472781623e-01
    4.471809626125602e-01     6.809494043743008e-01
    3.302159996030291e-01     4.585390314005313e-01
    5.748666723921246e-01     7.366600393945779e-01
    3.650916853640096e-01     2.551246590346664e-01
    5.268042886304959e-01     5.604884143856841e-01
    2.923293959354581e-01     8.036831985001504e-01
    8.150831896775994e-01     2.202335755261674e-01
    3.980713445250252e-01     1.688020676565091e-01
    8.831708129376681e-01     5.869117839618180e-01
    3.473647858329458e-01     3.081079929830097e-01
    2.915773010683311e-01     7.478360086136333e-01
    4.949109971873726e-01     9.558842711805680e-01
    5.225192362985787e-01     3.209450114810527e-01
    2.837108508245461e-01     1.786555004891729e-01
    6.898474044746844e-01     9.710924093757860e-01
    6.882594986026828e-01     9.219575420884156e-01
    8.629324407348099e-01     3.184609750500397e-01
    3.230375950721562e-01     4.949378296791719e-01
    2.168370584381146e-01     8.043023613298278e-01
    5.321532487196087e-01     6.594516346611801e-01
    4.528674783294329e-01     5.952344883401025e-01
    1.267664055498368e-01     7.944168650539172e-01
    3.524399263940962e-01     7.650475220967389e-01
    6.397122519668934e-01     4.581973304759228e-01
    4.723395999481101e-01     7.382809344499907e-01
    5.355659668470614e-01     5.965583931282874e-01
    6.971945080541765e-01     7.347902550781642e-01
    7.270855650434412e-01     9.012976976411131e-01
    8.377624175486467e-01     5.148940091229254e-01
    2.914339156422076e-01     6.778482687289455e-01
    2.965764292228742e-01     3.501196415403091e-01
    7.754312778603963e-01     1.098706630331830e-01
    4.910692365840382e-01     3.590821552173643e-01
    7.967026393066804e-01     4.008028595356776e-01
    5.787199708339371e-01     8.328857202079055e-01
    4.640629574537813e-01     6.585439873380704e-01
    3.079968876292425e-01     1.996560794771208e-01
    2.767057871684219e-01     5.599798179313038e-01
    4.523634967684491e-01     7.446991388247715e-01
    4.262399688113072e-01     2.192110466743654e-01
    3.404256214232226e-01     4.447635492619504e-01
    9.263676919375017e-01     4.850679347248414e-01
    4.764784830060883e-01     4.081106860820179e-01
    3.255465121194631e-01     6.844135374253408e-01
    7.231089123202185e-01     6.566484058826457e-01
    3.400635048222883e-01     4.882921970266397e-01
    4.079280927451616e-01     6.986424037000207e-01
    4.759195963922372e-01     4.569702894119650e-01
    3.551148839628052e-01     6.572724004649011e-01
    2.973222754187594e-01     1.257523175945532e-01
    7.169642632957201e-01     8.595464834709974e-01
    5.464101931380780e-01     6.679560106509511e-01
    3.173151150452944e-01     6.508012120454774e-01
    6.529927384877814e-01     9.056917628984690e-01
    7.090451671400108e-01     5.019450024528908e-01
    5.684167255174317e-01     4.937725449874570e-01
    1.192632698628251e-01     7.792099236823639e-01
    3.267057176033532e-01     1.293474235547446e-01
    2.430266534356266e-01     8.168393578732965e-01
    7.851321227019373e-01     2.278363997709332e-01
    2.821424228271269e-01     7.420962413324889e-01
    6.755288365883679e-01     7.087092674684107e-01
    7.051767472497164e-01     6.389705541733214e-01
    3.505947137189255e-01     1.725179871704792e-01
    3.330154032739013e-01     4.596196931480871e-01
    5.976572384756784e-01     9.965008480146740e-01
    8.142334904062214e-01     2.308592303902696e-01
    7.249244867512480e-01     9.917899820836521e-01
    5.109926941709982e-01     4.335976016190265e-01
    7.398171719031595e-01     9.417967524369796e-01
    4.121115976102628e-01     5.358106054429269e-01
    6.188795573298924e-01     5.331424995176087e-01
    9.327425262598144e-01     4.498580520786353e-01
    9.083878168809699e-01     5.756667204651368e-01
    6.964982461645196e-01     5.118872131605386e-01
    5.444472217561040e-01     4.320816093176352e-01
    2.972246986076265e-01     1.450149324793516e-01
    9.261885476152758e-01     4.021591957869373e-01
    4.347722210602905e-01     6.427316267295123e-01
    3.917498946303464e-01     2.580183708023445e-01
    4.410205546536324e-01     1.964854318281066e-01
    4.629883464439594e-01     6.737902733902076e-01
    9.374020432242476e-01     5.490326349933687e-01
    5.407630249176416e-01     5.043404353930285e-01
    6.442463334489569e-01     9.224062342637177e-01
    3.321264102881510e-01     6.494107343936935e-01
    2.352242991553259e-01     8.368024953836185e-01
    1.283558975090181e-01     6.544701782463310e-01
    5.155099241987022e-01     4.230855283724361e-01
    5.658456229150919e-01     6.713899705843815e-01
    9.567741967254566e-01     4.994222523032243e-01
    3.305865236953131e-01     4.896887015125758e-01
    7.999249492331058e-01     5.677877607459368e-01
    9.797414091165257e-02     7.806264454208384e-01
    3.484862040982910e-01     7.135162129401699e-01
    5.251197098876248e-01     9.301881692222159e-01
    2.727659132823063e-01     1.949586296045195e-01
    4.225288044918823e-01     8.527705121540230e-01
    5.634231481944352e-01     9.588318904158804e-01
    3.371194761966326e-01     3.715711909665277e-01
    2.902731967134626e-01     4.277103295362263e-01
    4.523773733608707e-01     6.636110908690416e-01
    6.833696660490467e-01     8.058160023952938e-01
    4.774020081901721e-01     9.725619088668860e-01
    7.796629231564394e-01     1.015450007232024e-01
    5.937887555059339e-01     5.266463235927485e-01
    2.237850735063263e-01     6.524109658518096e-01
    5.873512327379924e-01     8.197689300692248e-01
    3.389511540053985e-01     5.236958251758514e-01
    4.752745079210500e-01     5.445857187090125e-01
    5.191652764724898e-01     2.924475277199307e-01
    3.306073979296595e-01     1.040756094207158e-01
    8.210531066270966e-01     2.999289633624552e-01
    4.602799371212404e-01     6.127618560208105e-01
    5.590990038932426e-01     5.546714211412580e-01
    8.109281574792304e-01     1.476009572252825e-01
    7.452141503017226e-01     8.255579017495962e-01
    7.545625921923829e-01     6.811653709871028e-01
    7.956200677420846e-01     2.623920716008364e-01
    1.884974896368034e-01     6.427096473410883e-01
    6.012646835309537e-01     7.811107388550255e-01
    2.790389109647392e-01     4.828963561533015e-01
    7.708939497542644e-01     6.537742949781674e-01
    4.048948734382748e-01     7.277475951484857e-01
    4.811407090904968e-01     5.362618700191520e-01
    5.044201051978009e-01     2.994456905177216e-01
    5.878367525553408e-01     5.747858571142498e-01
    2.770723727361138e-01     3.595777139323120e-02
    3.202296947894351e-01     4.094116984442921e-01
    3.843503015102081e-01     6.305797631355244e-01
    8.329215935415043e-01     2.531078812458354e-01
    2.806558351807288e-01     4.256543399018281e-01
    2.644690949970728e-01     3.707737287095388e-01
    3.101721111055677e-01     1.744771588077922e-01
    7.871305574685472e-01     4.101624851350149e-01
    7.331748713561594e-01     6.822656723403844e-01
    6.297060101794302e-01     9.219673111894805e-01
    1.741037348618015e-01     6.476658403457908e-01
    5.306036707736952e-01     5.617024817301696e-01
    3.955264441212748e-01     2.505628013579178e-01
    7.592946807613156e-01     6.312767836733476e-01
    5.283924139667157e-01     3.448679272947277e-01
    6.411053787825264e-01     8.776849315761313e-01
    7.860345423462852e-01     2.359942260739581e-01
    5.696419890814182e-01     7.710163698449621e-01
    4.812089343367927e-01     9.001950877494688e-01
    1.477680873689183e-01     7.922209027934171e-01
    7.133523464207039e-01     8.053167880545300e-01
    4.095972581461774e-01     3.232015952334268e-01
    3.884034821585567e-01     2.344308941801447e-01
    3.859389600973673e-01     4.542773317070010e-01
    9.513449039069762e-01     5.060546565638752e-01
    4.122642168724612e-01     1.905046451320246e-01
    7.343818591758619e-01     6.492087286869858e-01
    3.827794771631237e-01     2.932054791223032e-01
    3.757180218692654e-01     8.365909793949264e-01
    6.218535817995577e-01     8.523836043020623e-01
    5.988243995493235e-01     8.328169626970944e-01
    8.071001151149751e-01     3.432261868657346e-01
    6.399903727650058e-01     6.608601497778884e-01
    4.309829092858779e-01     3.675226780026343e-01
    3.740780286568141e-01     2.499369989487047e-01
    3.335949143821565e-01     1.356903098209195e-01
    8.317877728820208e-01     5.382569780983233e-01
    5.010693126191886e-01     3.400466091674686e-01
    9.180361399185757e-01     4.410390275984485e-01
    8.315128450034140e-01     3.513598431699243e-01
    6.166247240132771e-01     5.695637269350354e-01
    6.337395153415915e-02     7.729450513121089e-01
    7.768887337191496e-01     2.542765843053458e-01
    7.477021165082368e-01     6.208462914899443e-01
    7.050750306916204e-01     6.731959482135019e-01
    4.526560719654380e-01     6.416221726622379e-01
    6.995505792796610e-01     4.593642429621762e-01
    3.463379280435618e-01     7.334888377743112e-01
    3.387108055903624e-01     6.480049455711515e-01
    9.141427814463735e-01     3.554681015934797e-01
    5.107199463514076e-01     7.492939223391480e-01
    7.945029195686681e-01     5.774348344435123e-01
    6.846915217317974e-01     8.427231617127773e-01
    6.456336353519891e-01     8.300292597411075e-01
    7.640236413524288e-01     5.069402376553489e-02
    8.860823589156701e-01     3.393924260826126e-01
    7.644456143425968e-01     5.236763002123122e-01
    7.003058252191269e-01     9.214056633042824e-01
    6.703968358042057e-01     7.104551122982842e-01
    4.628321786443997e-01     4.755039680986598e-01
    6.358442905450804e-01     9.183294370836427e-01
    3.567067901341661e-01     6.287587833693882e-01
    5.005206752190692e-01     5.731412025117484e-01
    3.337352777017869e-01     7.386348242096246e-01
    2.606819108616633e-01     2.375282318594196e-01
    9.638790320886378e-01     4.537452095688661e-01
    2.504635946761998e-01     8.725307934760884e-02
    4.249915810718966e-01     3.624362718582084e-01
    3.787306293839783e-01     2.666985370295909e-01
    3.708773779518175e-01     8.209880813116059e-01
    2.582805980094132e-01     9.276124965774413e-02
    7.321770257376891e-01     5.424940249957289e-01
    5.651997238643645e-01     9.411043103106088e-01
    4.997191830577647e-01     7.606879272802149e-01
    6.661351926735902e-01     8.903444463285133e-01
    1.922439223743819e-01     8.100477225104317e-01
    5.899487068278441e-01     3.512281414752890e-01
    3.916843973409142e-01     4.590317636101492e-01
    1.225549232786334e-01     6.487260803616606e-01
    4.001619526114103e-01     4.772949659731683e-01
    3.779307151018593e-01     5.893806406724950e-01
    1.433597986975492e-01     8.023376919825478e-01
    8.877129652900350e-01     3.991950071574386e-01
    1.439330495393699e-01     6.197758094219635e-01
    9.119732938772129e-01     5.120732130181888e-01
    4.775399658961047e-01     6.352617531340499e-01
    4.358412676635316e-01     8.083855299693311e-01
    5.127739403754129e-01     4.320917909207631e-01
    3.653496809735954e-01     8.081235189654247e-01
    3.447776663612968e-01     1.696700560051086e-01
    2.886317176075849e-01     6.228986612906491e-01
    6.473906337212668e-01     9.715619444668102e-01
    7.749055265471535e-01     1.777230469803882e-01
    5.056803551428924e-01     3.634655638919913e-01
    6.192808967988979e-01     9.031208081045359e-01
    6.019079013866607e-01     4.665036897107691e-01
    2.087310344077926e-01     7.638192871492000e-01
    1.179795653390993e-01     6.504324580546188e-01
    4.125781728151671e-01     7.757596090433917e-01
    8.495924331842061e-01     6.240372869043920e-01
    4.731548807331499e-01     8.007821482355618e-01
    1.959820861399983e-01     5.597104323357855e-01
    3.716146717759105e-01     5.439957507709962e-01
    4.078319292859588e-01     3.292985552163112e-01
    2.624199750984809e-01     3.531184976085742e-01
    4.671310548615470e-01     6.198934759552149e-01
    2.680628108788743e-01     7.414899195037066e-01
    6.285801027766296e-01     8.285551515891005e-01
    2.889233859407577e-01     4.237136550981820e-01
    5.742939203800507e-01     8.270494541853955e-01
    9.502566270562368e-01     5.363579404093975e-01
    9.400884970219626e-01     5.205433230695581e-01
    6.256356375742916e-01     5.561360146045059e-01
    2.869500127000947e-01     2.648074454416115e-01
    8.187691900165981e-01     1.715134542328353e-01
    8.717054994990682e-01     5.791794588355482e-01
    5.647621863849424e-01     4.492012075753067e-01
    8.511171007273045e-01     6.212685877411049e-01
    2.949680675735247e-01     5.180695950848814e-01
    8.744656745078541e-01     3.998285660670320e-01
    3.370967870417065e-01     3.109964950738918e-01
    6.087978566732170e-01     7.770511712294492e-01
    2.290916656336442e-01     6.907208534316798e-01
    5.657190764843968e-01     9.852969571721760e-01
    2.918588610342181e-01     3.399871095870513e-01
    2.427719976369928e-01     8.105548362316082e-01
    6.777310542114268e-02     7.786719537761064e-01
    7.479985124293448e-01     5.417776235360664e-01
    1.431640370399822e-01     8.152302886923530e-01
    2.310810759923556e-01     8.122650739889072e-01
    3.123219783564359e-01     3.470596495877230e-01
    8.038252469206006e-01     6.213011300954663e-01
    2.768973513387329e-01     3.245541927334650e-01
    2.604860091986374e-01     9.401651308340730e-02
    7.836754523306378e-01     3.792041838225847e-01
    2.875129761870183e-01     4.714585432686331e-01
    6.555720617681493e-01     8.754127435202390e-01
    5.063052459421478e-01     4.053364837706270e-01
    2.827753580146490e-01     3.815640408019475e-01
    4.555061110484195e-01     5.032833417930926e-01
    7.463491524398429e-01     8.968594052336527e-01
    4.813945967198298e-01     4.781536455264516e-01
    5.965189356531333e-01     8.965664863829900e-01
    8.460404676356487e-01     4.423940121575614e-01
    5.331021546555126e-02     7.530819907535087e-01
    3.931685754336379e-01     3.644270661891655e-01
    9.475536653185158e-01     4.574246949822577e-01
    3.169194378386105e-01     1.970007559612759e-01
    7.873758301130342e-01     2.056718645148089e-01
    2.786405107241544e-01     3.770073601788628e-01
    4.777086359464574e-01     9.820803858448028e-01
    7.815471521563263e-01     1.224498460123385e-01
    2.300479756096513e-01     7.166538945452028e-01
    9.327979528761806e-01     5.325738338834819e-01
    9.135382387932351e-01     5.855672613526122e-01
    7.022036503806839e-01     5.986107644351251e-01
    6.405283221728351e-01     4.757991888239966e-01
    7.717174965586739e-01     1.815773740109844e-01
    5.049225568990550e-01     9.838181482112881e-01
    9.080113658181673e-01     3.971785135820212e-01
    4.006477025186588e-01     3.277229478689007e-01
    8.994214708704215e-01     3.061828837759706e-01
    6.072878863429563e-01     6.203709519728312e-01
    4.804942954793209e-01     7.389752583954472e-01
    2.683094273032079e-01     1.012180306433886e-01
    5.200025675305160e-01     4.396357377332868e-01
    3.106916016848155e-01     4.244465669757309e-01
    4.944901559037473e-01     6.531889854854516e-01
    7.261679318259714e-01     8.804151603103038e-01
    6.398596590963117e-01     9.688627946437048e-01
    6.369446431244457e-01     9.233130328684175e-01
    8.221183022469226e-01     4.650899055152510e-01
    2.846355270789511e-01     8.819575988481637e-01
    8.123185730096675e-01     5.181901607811421e-01
    4.963868852482164e-01     7.588723843876377e-01
    4.376285900632645e-01     8.115958191818313e-01
    3.873270209546139e-01     2.229316039093213e-01
    2.678031931547550e-01     8.821282950223681e-01
    9.638297004430997e-01     4.535474545045964e-01
    5.551282207582418e-01     7.865789534204345e-01
    6.444569487733364e-01     4.799639251281412e-01
    1.330895362150707e-01     6.246634122684949e-01
    3.498284353358855e-01     6.565636071325998e-01
    3.009981301879813e-01     6.252925758425899e-01
    8.487012072722819e-01     2.818321173461620e-01
    7.568469488940879e-01     7.330885489797609e-01
    7.858259116682729e-01     5.477902720032329e-01
    5.722585342567457e-01     8.524505058064992e-01
    6.097012597162862e-01     9.096039875873093e-01
    3.625376127334221e-01     3.011952832639697e-01
    5.967253462182438e-01     9.770093081001280e-01
    6.258968380832828e-01     5.179869878866722e-01
    4.728271220300670e-01     3.385503812638502e-01
    5.195849437220839e-01     2.769450938569552e-01
    3.253815251811230e-01     3.345296259591363e-01
    8.749804043306361e-01     6.086438520758298e-01
    7.733080709321327e-01     4.996485635670572e-01
    4.004190034762733e-01     4.382865883228130e-01
    3.450202932504103e-01     3.165990732694721e-01
    3.020642024598589e-01     2.677530706488717e-01
    3.895227907333814e-01     8.859778977251521e-01
    5.772531964389432e-01     5.973805273186820e-01
    7.691363848513120e-01     2.338209996071522e-01
    6.634652296386552e-01     6.264385800420976e-01
    5.900394364700716e-01     3.512992249903827e-01
    4.502269323274620e-01     6.849584613209209e-01
    8.708483267615957e-01     4.852935465534252e-01
    4.848202659788743e-01     7.264983159093926e-01
    5.234366883800698e-01     3.328650504705066e-01
    8.029173785536129e-01     5.569564518335969e-01
    8.023591407635934e-01     1.322063710872606e-01
    6.891632739542919e-01     5.587742704059125e-01
    7.695932333308872e-01     6.351631807368515e-01
    7.528416453274815e-01     5.281465575496116e-01
    4.065135063332567e-01     2.517383185423712e-01
    7.438809622074368e-01     5.917228779566660e-01
    6.300248361074201e-02     7.578533254988207e-01
    8.371054028343607e-01     4.383068423321682e-01
    4.060929986142625e-01     6.305395476302456e-01
    7.671268136059763e-01     1.179868265679046e-01
    5.034521051399835e-01     4.555318810010700e-01
    9.020450927129144e-01     3.129349017070240e-01
    7.217472212236993e-01     5.019627329680835e-01
    1.934387125223341e-01     7.777154844196503e-01
    7.525447312945781e-01     4.477893177465334e-01
    3.567891482462645e-01     4.342940897734029e-01
    5.357143821223548e-01     4.979988588232153e-01
    4.158658658387661e-01     7.436492686517340e-01
    7.940959617508595e-01     3.043756173014575e-01
    4.263073896212097e-01     5.367911672777960e-01
    5.698588463592618e-01     3.939995476624898e-01
    6.017404928454900e-01     9.583631278324358e-01
    2.123897849792514e-01     5.995830933788618e-01
    1.729080612996888e-01     7.003627083329077e-01
    7.678994643885700e-01     3.174593349476059e-01
    5.664702247785163e-01     7.547280426008383e-01
    2.499392493196766e-01     8.257256080336837e-01
    8.441683151986542e-01     3.896192462298596e-01
    8.623850715444568e-01     2.764469232543515e-01
    7.646759451995717e-01     7.436700175258171e-02
    5.633554557752309e-01     5.500843615690209e-01
    7.845912534379950e-01     1.543138877469702e-01
    5.080110448053803e-01     7.553311111083715e-01
    2.317881738914497e-01     5.219749275814360e-01
    8.592653821241135e-01     3.722500868511266e-01
    6.396085251612990e-01     8.865333514706142e-01
    8.961238393676927e-01     3.257335822671503e-01
    8.492754360926601e-01     5.245444899221329e-01
    7.174144579026653e-01     6.749500373610576e-01
    4.387693527583155e-01     2.101099536899698e-01
    2.837096470972836e-01     7.533951058764127e-01
    2.967632776082595e-01     5.040363446751588e-01
    4.907057158006617e-01     8.158726581483586e-01
    5.028805975825117e-01     4.872130597571029e-01
    8.133458962256633e-01     4.463368400819261e-01
    5.337947574362834e-01     3.441694566367253e-01
    7.550603045941796e-01     1.711436618849937e-02
    6.977428175312533e-01     6.629089768962569e-01
    6.330853143284055e-01     8.703349308588103e-01
    6.268577251177423e-01     7.138828957258656e-01
    2.858082947507178e-01     3.635806149699481e-01
    5.871273549672928e-01     3.413168388471374e-01
    7.468305166465717e-01     6.556025448885512e-01
    2.583986393234706e-01     3.587592421339341e-01
    7.590408262458241e-01     5.612782486787570e-01
    4.594283117714965e-01     9.500330368597282e-01
    3.453612572523805e-01     6.523478206731459e-01
    4.051540351912393e-01     9.138441163378447e-01
    7.226045416865666e-01     8.678397774296134e-01
    5.780715405982778e-01     9.833433108396353e-01
    2.541536634423889e-01     4.915447484437266e-01
    5.839027334393391e-01     6.785838584820111e-01
    6.423928681222333e-01     8.955168579588988e-01
    3.769384830635459e-01     1.741861335216902e-01
    5.878298995374650e-01     7.399878452075551e-01
    4.914497457019728e-01     5.522256415297935e-01
    5.663910851002100e-01     8.244255431080904e-01
    4.371290439017130e-01     3.201908418780253e-01
    6.088001077733626e-01     4.183274820289560e-01
    4.173559598656814e-01     3.879973781584128e-01
    2.948777154519716e-01     1.768206504740990e-01
    2.509408448598353e-01     6.539093429815241e-01
    2.555516454657415e-01     8.292750305248164e-01
    6.674519779097381e-01     6.115370175376815e-01
    4.000450414045965e-01     5.639753390292503e-01
    6.727177283019606e-01     7.817810405001993e-01
    8.893971602501889e-01     5.317422614941745e-01
    5.079108697537600e-01     7.068495023354833e-01
    7.167291518276652e-01     7.492787715565100e-01
    5.012255269502280e-01     5.622730152816404e-01
    4.399056747903030e-01     4.842081039763574e-01
    3.276693068692504e-01     3.397129011437802e-01
    5.556819970694464e-01     7.534293902650919e-01
    4.718125237566972e-01     4.858725362519134e-01
    2.417309229150132e-01     6.775096913335606e-01
    6.098354204405699e-01     3.887316387375800e-01
    8.786482373484439e-01     5.303012723352351e-01
    4.561453468319379e-01     8.782375907650473e-01
    8.011173514625477e-01     1.245357966783887e-01
    3.058125619273564e-01     1.977804846061419e-01
    3.070256018977268e-01     1.722799362039298e-01
    2.501531616226879e-01     7.796526826328266e-01
    5.886011725982812e-01     5.656064396380365e-01
    1.700783873792038e-01     7.996671250934971e-01
    4.442844304708502e-01     7.515940754819018e-01
    5.856217302123797e-01     4.283083565851498e-01
    7.597876205103583e-01     6.553108003259603e-01
    3.817222535981507e-01     2.231744783136820e-01
    6.904011089288323e-01     8.368050980649410e-01
    5.834702416447726e-01     6.248724156296050e-01
    2.083412817165515e-01     7.141130241714745e-01
    7.925130870760095e-01     3.587240792654980e-01
    8.454350086410641e-01     4.689781740605612e-01
    7.952515945074488e-01     1.252963135116503e-01
    4.552850997290296e-01     7.779351915656810e-01
    5.588372728355365e-01     3.620595673665404e-01
    7.947265233129165e-01     1.660840414473682e-01
    5.687424621434017e-01     9.096186522174623e-01
    7.918888088339098e-01     6.658052580706899e-01
    6.501451668081067e-01     4.315727402374356e-01
    7.048112858099578e-01     4.827424898249085e-01
    5.816385889314202e-01     5.426949938322331e-01
    3.034999329181135e-01     4.513995107017856e-01
    4.481153320434863e-01     3.816533908206137e-01
    2.628521201014546e-01     2.349952901205238e-01
    4.813202335534057e-01     2.890652875294099e-01
    1.428275750301456e-01     8.036172455253192e-01
    1.651914594608214e-01     8.163755635033113e-01
    8.328429858472782e-01     3.867066963751692e-01
    7.023043650759195e-01     8.206082638227900e-01
    4.231647159352377e-01     9.059148616741185e-01
    6.749060150051677e-01     5.588034456551597e-01
    5.516224155541828e-01     7.937676425017669e-01
    4.399404056566197e-01     7.377821787317094e-01
    4.510017236448686e-01     2.615125314535509e-01
    6.970011369369652e-01     7.487666431370895e-01
    4.002446231864751e-01     3.628275978547582e-01
    6.462658033211831e-01     4.780408350020662e-01
    3.826995945741417e-01     4.765646246638012e-01
    4.090306928778045e-01     4.219037137983949e-01
    4.253509303942016e-01     5.498451607546093e-01
    8.128855088669241e-01     5.537199165138083e-01
    2.845285860787469e-01     6.176465635066491e-01
    3.071034726591652e-01     5.613022115003786e-01
    3.160829858909604e-01     6.721149947120797e-02
    5.987962648676159e-01     9.332648221910971e-01
    3.465606936762123e-01     7.772586555436035e-01
    4.503839858646800e-01     6.466126537266083e-01
    8.637309900439908e-01     2.796430999521237e-01
    3.321340031369177e-01     5.147035033480420e-01
    6.680564618688830e-01     7.080904228155490e-01
    7.916739857853438e-01     6.743673260420356e-01
    4.612893202704892e-01     9.744835545687072e-01
    7.158976480825007e-01     5.801060479502014e-01
    4.165242649987789e-01     6.768746896441321e-01
    8.116419237696257e-01     2.080962808871158e-01
    5.020502567178606e-01     6.101012647262332e-01
    8.959642623676367e-01     5.269891458849412e-01
    9.081755257832111e-01     4.700700318235290e-01
    2.766936923235063e-01     1.504823766092472e-01
    9.276712827146679e-01     4.797655149748268e-01
    3.903345410092028e-01     9.091947165773381e-01
    5.430098764298686e-01     4.421769642590894e-01
    7.474504868050963e-01     6.541006676925294e-01
    5.171217319176151e-01     2.980328533359136e-01
    4.169037866125258e-01     8.877819930133238e-01
    5.004919721985074e-01     7.922691864566057e-01
    5.756931440556887e-01     5.215449273416869e-01
    7.266230095343874e-01     6.526926666108989e-01
    5.888228680586222e-01     5.932886213602366e-01
    5.244560358658806e-01     4.690610764868506e-01
    4.190225148815128e-01     8.141187330992302e-01
    5.705838580788295e-01     4.704927943831508e-01
    7.798160192778787e-01     2.339807668841034e-01
    3.556803660687811e-01     1.768304659841036e-01
    6.081205133802713e-01     4.718157881307249e-01
    8.721550681494762e-01     5.936134105113926e-01
    3.555195527294517e-01     2.227949133247550e-01
    8.084348283225783e-01     1.958629395333070e-01
    1.864161692414793e-01     6.540043953213175e-01
    8.624189479964248e-01     3.758970288857231e-01
    7.109275024127525e-01     5.544722175476753e-01
    4.277941757490538e-01     9.183576774538685e-01
    6.877546433203158e-01     8.791706382834875e-01
    6.105302849229884e-01     6.787155125970572e-01
    2.606221656291289e-01     2.530224547153124e-01
    7.527837760380059e-01     3.709047855641150e-01
    7.693204371305989e-01     5.480649053781851e-01
    3.209619964459832e-01     3.831443730240056e-01
    5.038669803979418e-01     6.652334376902499e-01
    5.514235926998409e-01     4.021654847132666e-01
    3.787478846131813e-01     6.997338201296489e-01
    7.821432302235435e-01     3.498735349190596e-01
    5.365096406886444e-01     9.591919483726431e-01
    5.499998106864247e-01     5.947229454093064e-01
    7.574849108525156e-01     5.731938503646691e-01
    7.636378862610745e-01     4.604443025239182e-01
    7.508926426121217e-01     4.743208157298298e-01
    6.219378100256533e-01     9.094757361219616e-01
    4.367683127837796e-01     4.193104775791072e-01
    1.598440342442402e-01     7.462172964977135e-01
    3.501446659624953e-01     5.699090959227154e-01
    8.443116564225859e-01     3.197889469243115e-01
    6.365520295940210e-01     7.538386033495578e-01
    5.919569506159906e-01     6.769259248933488e-01
    4.338505343624858e-01     3.158569019843905e-01
    5.345115992795134e-01     9.880868369891735e-01
    3.527234137697673e-01     3.552041062654394e-01
    2.106383869364473e-01     6.190202009741643e-01
    6.717318276154653e-01     5.769963063313401e-01
    7.937556818178751e-01     4.789134844450595e-01
    3.824103309152393e-01     2.419273519712589e-01
    6.178484995432607e-01     7.532548484311912e-01
    2.484388682703461e-01     5.552863710520792e-01
    4.490230052035555e-01     2.028514842649624e-01
    4.464701431957129e-01     3.602568001608963e-01
    1.060586928360737e-01     6.921917683760987e-01
    7.625430290611502e-01     2.667820258630246e-01
    7.136772271921727e-01     5.972940988856632e-01
    7.615004937706548e-01     4.435256424893181e-01
    6.788961955814183e-01     8.296383331834767e-01
    5.187671316416899e-01     3.013662703323714e-01
    5.256175460171892e-01     6.275307384424752e-01
    7.157655875427120e-01     7.193524788195644e-01
    7.370421288330363e-01     5.285599543686246e-01
    4.152160988353389e-01     2.093766249765151e-01
    2.092386172029718e-01     7.104895381810707e-01
    2.652183773723186e-01     6.614128831256550e-01
    8.014413457360492e-01     6.476756152326563e-01
    1.825046189776663e-01     5.807244920117504e-01
    2.935626173571408e-01     4.687454845191321e-01
    3.227310109308232e-01     7.342615423063480e-01
    5.150276170424677e-01     6.370020986824387e-01
    7.146247187191623e-01     4.968454312858841e-01
    8.965988027482769e-01     4.722248092485245e-01
    7.210249529143414e-01     4.874532200744319e-01
    6.159734934642793e-01     7.424519318888461e-01
    1.235523887594364e-01     7.553815059461503e-01
    5.717414831828508e-01     7.365707984500609e-01
    2.917403773463789e-01     2.734722055270986e-01
    7.124370129730104e-01     7.279204252565118e-01
    5.147223419455951e-01     7.622686693627214e-01
    8.601493635540650e-01     4.817215001881915e-01
    3.071217239338845e-01     7.213019954481542e-01
    5.074299007211736e-01     9.970921045358004e-01
    4.100659424236467e-01     3.237828761890023e-01
    9.044119387800155e-01     4.230332502162608e-01
    4.560070069114882e-01     5.367080447732242e-01
    3.970098198594737e-01     8.353010899774265e-01
    8.045737397758038e-01     1.864607648330953e-01
    7.644724516973185e-01     9.628647270056856e-02
    5.053976224590444e-01     5.498466658043298e-01
    9.272197274908739e-01     5.027386647102106e-01
    7.830628007293809e-01     2.003901648803182e-01
    6.770935213632649e-01     7.413087827520445e-01
    2.015550637171800e-01     7.034532525505652e-01
    4.664721570915593e-01     7.591958172984740e-01
    8.795881274329419e-01     4.725840642830647e-01
    3.523773915381591e-01     9.057468098897083e-01
    5.391771209499024e-01     7.624054614239006e-01
    9.328634391453441e-01     5.116613406505531e-01
    6.143022824147931e-01     4.923018672526275e-01
    3.130588159320652e-01     8.984310851857906e-01
    8.230671104380967e-01     6.087167058751449e-01
    4.195249213756644e-01     5.171456687531663e-01
    8.570209222539888e-01     5.218223825494437e-01
    7.910619918668156e-01     4.397068847256959e-01
    4.391162377401751e-01     3.357525156305148e-01
    6.781837620829486e-01     7.245892992664134e-01
    6.123394330162655e-01     8.458343743739878e-01
    3.126722743563377e-01     4.015629489259415e-01
    5.406950601863103e-01     6.351759046053596e-01
    2.508491823081187e-01     1.542639322344777e-01
    3.825819799734681e-01     5.684565295970210e-01
    4.860247574687306e-01     6.277838593732400e-01
    5.677153263679452e-01     9.402392184635021e-01
    3.831093013099141e-01     7.034685768755439e-01
    2.959015462734471e-01     4.246735295897301e-01
    8.961695949393562e-01     4.098950104449501e-01
    6.068849794539808e-01     7.863867692118928e-01
    2.679159520275066e-01     4.925472450692749e-01
    3.210444756295945e-01     3.432642648646285e-01
    3.545522472427352e-01     3.152063384848575e-01
    5.596395712971607e-01     7.488711194333043e-01
    4.491890773651563e-01     7.121594068601192e-01
    2.600503467280544e-01     1.317312013181495e-01
    4.593574699428793e-01     7.133237168481191e-01
    5.596088815220942e-01     4.762995106605968e-01
    1.961540193372487e-01     8.026534848848430e-01
    7.544817708872139e-01     7.445292198368324e-01
    7.432481129914201e-01     6.197667760968636e-01
    3.416954824207999e-01     9.115572521237042e-01
    2.931081333530251e-01     6.457673238245442e-01
    3.839234759470276e-01     6.469181055072364e-01
    4.643903246435855e-01     2.147573269868609e-01
    3.739935428362472e-01     2.440485830269190e-01
    5.399715120468691e-01     4.403569493892658e-01
    5.098843811661469e-01     5.558986213180146e-01
    4.056084394085605e-01     8.651719411463008e-01
    4.675113271485936e-01     9.428898453024388e-01
    3.474643626033188e-01     7.608825900163204e-01
    4.738940097206362e-01     4.698111949974981e-01
    7.466303087254572e-01     9.378230898909734e-01
    2.481451165271253e-01     8.217282793980715e-01
    4.468863413251839e-01     7.459072809564427e-01
    8.197284980006816e-01     5.482286659302996e-01
    4.678932828102454e-01     9.800267787487253e-01
    6.657619997287187e-01     9.212656404222412e-01
    8.522624645578613e-01     2.274612183118321e-01
    3.607117524347069e-01     4.883536031495258e-01
    5.818627480843304e-01     7.006734096464902e-01
    5.507152623461193e-01     4.826154019916810e-01
    6.309732834418854e-01     6.325415534319114e-01
    5.176939521783759e-01     6.957934093494383e-01
    1.127930993778289e-01     6.918138319386715e-01
    6.379835955277234e-01     9.390279180853457e-01
    4.707846636872572e-01     8.181596094202860e-01
    4.646713872072280e-01     2.218713710216615e-01
    6.281465270160144e-01     8.924826964867683e-01
    4.205026954212693e-01     9.321407411158709e-01
    5.305723253194907e-01     9.725125557042817e-01
    2.501744200551158e-01     8.190277474817699e-01
    4.067840086391448e-01     3.495742197501133e-01
    6.100475840075066e-01     4.363531838583317e-01
    4.830620991130311e-01     3.721155383075654e-01
    8.818430174553447e-01     4.176900965053487e-01
    6.510971915502539e-01     9.388366490623405e-01
    4.458208552896363e-01     4.823277630035543e-01
    4.968220703932549e-01     7.097453442932753e-01
    4.948845296289532e-01     8.160099155418131e-01
    7.689376895907960e-01     1.355889043800842e-01
    6.062121063605710e-01     6.598432050045786e-01
    5.641096222550076e-01     4.344823786671218e-01
    1.250939927236769e-01     6.477207220354995e-01
    4.078355311989765e-01     8.746455819516276e-01
    4.864725027276954e-01     7.204985829833465e-01
    7.700659619248792e-01     1.558895666043988e-01
    8.318515443502140e-01     5.807915238590937e-01
    4.767036473995023e-01     4.069201179152818e-01
    4.627309397119099e-01     5.697885027355087e-01
    6.265497531394550e-01     9.083672235792974e-01
    8.236108735240777e-01     1.940174491688389e-01
    7.578089630179534e-01     7.168350049735969e-01
    5.245664344775656e-01     9.797295400677773e-01
    8.065466595259466e-01     5.976814324552685e-01
    6.710440564162639e-01     6.214282639206380e-01
    3.764196283816221e-01     6.748524219312647e-01
    8.654707523488118e-01     2.881648537336705e-01
    7.948006787802490e-01     5.448247170443492e-01
    2.730055933459808e-01     3.887758642033473e-01
    2.347965669952872e-01     8.361823158408358e-01
    4.393539270905350e-01     3.322181073955980e-01
    5.509326518024735e-01     8.505970988278138e-01
    4.571016170468672e-01     3.474556596231684e-01
    4.097947877729585e-01     8.827240092801422e-01
    1.713075170708825e-01     6.460326677708610e-01
    7.667323777262868e-01     5.233405429108599e-01
    6.581007655866867e-01     9.310547804886199e-01
    2.060761710427086e-01     8.061874049315637e-01
    3.620468145656950e-01     2.278347594090748e-01
    8.510541341511244e-01     3.324281072601266e-01
    5.145808953614432e-01     8.556342594674968e-01
    4.558815710688510e-01     6.643562514965714e-01
    6.271992340337309e-01     4.482738662047752e-01
    3.746622294602361e-01     4.785815145438657e-01
    5.558026844911595e-01     4.886439659702766e-01
    8.555528650472726e-01     4.287636034737337e-01
    6.220037606801514e-01     6.813394812798893e-01
    7.048384273648697e-01     4.850112944840704e-01
    7.876379237126495e-01     6.001214833542425e-01
    8.860255205755643e-01     5.093847153815698e-01
    5.920732316471466e-01     4.631387918739178e-01
    4.975369715075083e-01     4.667064101787793e-01
    4.918303105513243e-01     6.849878523401194e-01
    6.177306473608463e-01     5.234274462226302e-01
    3.482841353027329e-01     8.659237153332036e-01
    2.453114671144059e-01     8.196625912493699e-01
    3.205261037454189e-01     4.307449147809799e-01
    6.322867339638819e-01     7.825796291058792e-01
    7.874818678865831e-01     3.692213458748123e-01
    3.031823620076965e-01     4.589232535066929e-01
    3.654671147809327e-01     7.995819892617113e-01
    4.926824964456656e-01     5.706329916138536e-01
    3.610942534438135e-01     1.259252806388640e-01
    3.773537547602400e-01     6.979424781373617e-01
    7.939308930798610e-01     6.849519700270180e-01
    5.905934959802189e-01     8.418127346038691e-01
    5.980310408313297e-01     9.044859935172491e-01
    3.536560301156548e-01     4.738229939977590e-01
    8.348446778641438e-01     1.828029940927323e-01
    4.260583865690737e-02     7.340600709533478e-01
    7.677653442999823e-01     7.216872718929828e-01
    3.036456455684625e-01     7.251010965847789e-01
    2.822306358903310e-01     1.871857187064641e-01
    7.113352545677812e-01     9.585620462473809e-01
    7.546022798943023e-01     2.436938972262989e-01
    5.650149061814975e-01     3.914443421726824e-01
    5.652480065532701e-01     6.573398432947543e-01
    7.327371371137648e-01     6.207023545681860e-01
    8.729761548964584e-01     3.819763182496149e-01
    7.893195064536837e-01     2.814106873094108e-01
    4.911926153473277e-01     3.971059398230286e-01
    5.735639366684264e-01     7.417736574646464e-01
    8.136251246410812e-01     4.366895148013071e-01
    4.254393437711737e-01     7.182248124492296e-01
    2.820489429816735e-01     2.044905627856474e-01
    3.945824709881457e-01     4.816055303369624e-01
    9.220193159265339e-01     4.831837588702346e-01
    8.532960441131814e-01     4.971432416288835e-01
    8.091737638455174e-01     3.774785722681439e-01
    5.915995463054370e-01     6.048593466384513e-01
    7.398313768392467e-01     8.110815888561370e-01
    1.371729880805163e-01     7.940549057187021e-01
    4.393564480677042e-01     2.558190701670998e-01
    5.336566654142820e-01     5.174580781404374e-01
    3.210106309879407e-01     3.838503981780814e-01
    7.758490143452004e-01     9.873665573274293e-02
    3.538977980109820e-01     2.518473245834690e-01
    2.937006231075513e-01     8.098419771635115e-01
    4.879390071116103e-01     9.006427828916830e-01
    7.626165586376704e-01     6.400096543804084e-01
    8.058983238872908e-01     5.449493053883712e-01
    6.445694730421444e-01     6.822925848309332e-01
    3.128384694796965e-01     5.085369233668483e-01
    3.936691509770015e-01     4.573667144125870e-01
    3.470218943842944e-01     8.928025007945583e-01
    1.802300297898011e-01     6.619358733993091e-01
    3.936622604867878e-01     3.543499270088113e-01
    7.273979154103745e-01     8.882225788294398e-01
    7.518973136732224e-01     3.903900658692692e-01
    3.545117587744446e-01     4.163424898457007e-01
    4.866141173107119e-01     4.310668326283964e-01
    4.247161748008210e-01     6.325744002736932e-01
    3.445929569055748e-01     5.120334037470544e-01
    4.095179264316235e-01     8.918212257932793e-01
    7.764193754949209e-01     2.117949745566544e-01
    6.836844020763184e-01     9.610612439661208e-01
    3.365363382983272e-01     9.910831341004633e-02
    9.314912903015020e-01     3.790949242287871e-01
    9.703869809927870e-01     4.869036699269319e-01
    5.243449183199617e-01     6.770342181120079e-01
    3.362278274871020e-01     3.769756975550279e-01
    2.791153821595912e-01     6.149733409342127e-01
    8.930963853308002e-01     5.096708698213193e-01
    2.840694452946161e-01     7.245451163964811e-02
    2.966761249307702e-01     3.943401999381455e-01
    7.681685002793065e-01     1.874661015818121e-01
    4.280636769469872e-01     4.021362433315141e-01
    3.102419235033792e-01     8.015499072912318e-01
    3.097290463611346e-01     1.757515602598940e-01
    3.663514077625596e-01     8.122197170549705e-01
    5.553668223452253e-01     3.371113946484848e-01
    4.932705425859508e-01     8.718325112948502e-01
    5.930396072847303e-01     7.945389910877835e-01
    8.532200752643354e-01     2.083105885131095e-01
    4.604309976730904e-01     9.750141663077766e-01
    5.321014427039313e-01     3.591148441123948e-01
    9.840857579019235e-02     7.951814875574017e-01
    4.399659762496536e-01     4.226237120548841e-01
    4.373895336433969e-01     5.329288467674009e-01
    8.829837407956914e-01     4.878302569055919e-01
    2.566421183852529e-01     4.839910977824380e-01
    5.232079232393676e-01     8.176603836765546e-01
    3.879224621575954e-01     6.376973956068421e-01
    5.651988022749924e-01     8.626040817718160e-01
    3.730157544202750e-01     6.717931441841606e-01
    6.368607799303964e-01     7.470320488529252e-01
    7.380542954990945e-01     9.566092665852602e-01
    2.698940274228858e-01     6.823051250402619e-01
    3.852300059035190e-01     2.659557882772208e-01
    3.984653229202765e-01     5.020204751100279e-01
    8.264695591250084e-01     3.655441773784717e-01
    3.214170322363650e-01     8.773472610664573e-01
    8.698108045073479e-01     3.750882031667386e-01
    7.743462596819745e-01     3.459480660948309e-01
    3.303975297972778e-01     3.647321928100749e-01
    5.070872695583553e-01     6.860848206512854e-01
    5.776353921817516e-01     9.277499784704331e-01
    4.530632027849550e-01     8.390444424413022e-01
    1.746969402391385e-01     7.450314948404789e-01
    5.313318406308578e-01     7.780019177490712e-01
    7.108381107306537e-01     8.950704796470951e-01
    1.810218323670004e-01     7.670354729552751e-01
    3.656134317656263e-01     6.627019909126669e-01
    2.528566293540661e-01     6.537060347005667e-01
    8.115606526418186e-01     3.228729336312905e-01
    8.116966459371551e-01     3.746110586605045e-01
    1.969194930713060e-01     7.373676474629741e-01
    2.643019939229581e-01     8.784961337378233e-01
    2.923440612107537e-01     3.797632886759550e-01
    2.361603738177377e-01     8.631495517888442e-01
    5.438189615242180e-01     6.789239294654787e-01
    5.508146039950652e-01     7.291151357744408e-01
    3.179094935813261e-01     4.197573274464823e-01
    3.183344582814347e-01     7.246041890569642e-01
    2.609171770103428e-01     4.128650357766044e-01
    2.531303708788354e-01     6.700890393221809e-01
    4.654703274931656e-01     2.852591956148709e-01
    3.539801243599687e-01     1.317241398836152e-01
    7.811928292688649e-01     7.027967235665777e-02
    2.768564880428314e-01     6.147816378960608e-01
    6.833663620611301e-01     6.302197983471675e-01
    5.398898913795989e-01     4.261736642759596e-01
    3.842713390442948e-01     7.931421689138495e-01
    4.585375663185283e-01     7.758049382838192e-01
    4.613842338257543e-01     7.575164414606977e-01
    3.096177227620122e-01     6.087804944539963e-01
    2.202057798802749e-01     8.056571303983988e-01
    3.449052923776866e-01     4.075234667251496e-01
    2.912803336645159e-01     1.330623789290011e-01
    3.794873901143719e-01     7.232348116323609e-01
    7.614041299217315e-01     3.384417744663323e-01
    6.400374367648570e-01     7.197772947189516e-01
    2.554269809369261e-01     8.263168260115160e-01
    9.191211469519145e-01     4.191319129484111e-01
    4.968532475816153e-01     5.667703444765737e-01
    8.837671135461571e-01     2.797038095647604e-01
    7.618360893949818e-01     5.647663053439783e-01
    3.277375627818069e-01     5.374301387383976e-01
    8.744721935991768e-01     5.201338593169447e-01
    7.492277824135828e-01     8.748479427607391e-01
    1.944205545118376e-01     7.856360788546357e-01
    7.500794535820037e-01     3.719518259240262e-02
    7.400355052466675e-01     8.465677377754747e-01
    6.719061623173338e-01     5.759719350038709e-01
    8.423992093578748e-01     2.896337511145493e-01
    6.842947072900210e-01     6.613225056865412e-01
    5.863544606149841e-01     6.119694352045951e-01
    3.997944565186059e-01     7.230755631039358e-01
    5.659400678404587e-01     9.205993613623912e-01
    3.158337439716592e-01     8.023974953732890e-01
    5.653200389460991e-01     8.110714951678178e-01
    6.906244512389736e-01     8.810601412099538e-01
    4.544671297719732e-01     2.180053465300817e-01
    4.221159239260819e-01     8.580416049173396e-01
    5.165391519889941e-01     3.477177854477060e-01
    4.385875907072496e-01     6.819356129859644e-01
    8.461632651504486e-01     5.991602838420323e-01
    6.640327510866838e-01     6.373336701362832e-01
    6.547944346491921e-01     6.641458722213595e-01
    6.666610858131635e-01     8.260796319485728e-01
    3.307093084399667e-01     1.364012543748887e-01
    2.547917804981977e-01     1.310822662554814e-01
    7.909151716177608e-01     1.564563885240267e-01
    3.662818650752510e-01     5.941993872640176e-01
    3.533935385993701e-01     3.732336777770886e-01
    7.070232024140255e-01     6.241492813322399e-01
    4.027126890400177e-01     9.340557023233206e-01
    6.159751250640939e-01     6.438700360300200e-01
    2.189032771393956e-01     6.191043222344498e-01
    2.724970128115792e-01     3.056275428514538e-01
    8.972817491247075e-01     3.382196736374312e-01
    3.775539031022906e-01     8.687443403896455e-01
    2.522882585992240e-01     7.935635255630562e-01
    4.599682628233203e-01     7.907458950100639e-01
    4.531115494428751e-01     6.081631646048781e-01
    7.495747219048293e-01     5.615228567849240e-01
    9.481734934966936e-01     5.088848573238373e-01
    3.280775881076939e-01     7.617234367326958e-01
    2.503789298681041e-01     4.037714170924817e-01
    7.098259284328742e-01     9.021080264565947e-01
    6.379035861966192e-01     5.577581767523412e-01
    2.595185319341068e-01     3.221572296528562e-01
    1.874176703476406e-01     7.995859872159659e-01
    9.905904669753796e-02     7.740600321676603e-01
    5.632505029915594e-01     4.767448765507059e-01
    5.407769831706531e-01     4.734882566118120e-01
    4.214927610103363e-01     5.376879279940717e-01
    4.663310151519062e-01     5.941992828832712e-01
    1.047846588366358e-01     7.316427083276120e-01
    1.395974513984845e-01     6.282576701420024e-01
    2.824335927115278e-01     2.498603515153125e-01
    2.908625319312600e-01     2.387004429052372e-01
    7.763334637543049e-01     5.521541918267759e-01
    8.929950389054591e-01     4.519982351826168e-01
    8.133540449155883e-01     5.097842059576573e-01
    4.580424511790326e-01     6.768416361250101e-01
    4.335428947486708e-01     4.964891153142775e-01
    3.883122561149699e-01     7.357613288011113e-01
    3.002118844862826e-01     2.633299381788449e-01
    1.819077501820495e-01     7.760684024780192e-01
    8.983589048437080e-02     6.850203920694400e-01
    6.305086762678923e-01     9.311562289546592e-01
    2.848215873360075e-01     2.555674564333038e-01
    8.526006881187608e-01     4.524756638866257e-01
    5.908509733711743e-01     9.821698695232713e-01
    2.986200952154811e-01     1.036607787698320e-01
    2.239554530258669e-01     6.700688248656672e-01
    2.775278822650673e-01     7.803491442010867e-01
    3.258505754301868e-01     3.190632729471097e-01
    5.209193165226907e-01     8.177283441721243e-01
    1.765599709901547e-01     8.224236098434745e-01
    7.022566983344931e-01     5.398414359607537e-01
    7.114540565784570e-01     6.950050933525228e-01
    5.854616067577161e-01     5.604332788219130e-01
    2.921010854007192e-01     6.826576430394713e-01
    4.921275017581969e-01     5.858183373496869e-01
    8.439346667467008e-01     2.950272385738510e-01
    6.769580219952211e-01     5.028812978055334e-01
    2.638125251920017e-01     2.427467759063425e-01
    7.648154641757029e-01     1.370149988591460e-01
    2.792934402300618e-01     8.473499842949950e-01
    5.432764405672444e-01     8.397215127637631e-01
    4.750140693439244e-01     2.371664238335937e-01
    4.960252662627573e-01     5.188348894079565e-01
    3.548930431187359e-01     5.769647383889853e-01
    5.248445005119990e-01     9.764667224229830e-01
    8.843270812176625e-01     5.148828542278302e-01
    6.456922080947803e-01     6.512251878553237e-01
    6.579568471221551e-01     9.719387206945674e-01
    5.636932543581916e-01     8.969173479468412e-01
    4.842329270566448e-01     2.977073176820706e-01
    8.146086797930641e-01     3.154935920991516e-01
    8.958445208394461e-02     6.875203727764591e-01
    7.168718304799970e-01     5.142106069523225e-01
    4.118757481973371e-01     5.181463433235988e-01
    7.964132838079914e-01     4.499991205066761e-01
    3.284439426370571e-01     4.053068825604181e-01
    6.274399261584154e-01     7.735208350456447e-01
    6.988882189765787e-01     6.309328183342660e-01
    6.396428566275100e-01     5.180311039982647e-01
    3.224626018580410e-01     8.505635485955765e-02
    3.326189699540819e-01     6.481067199510802e-01
    4.547169637959895e-01     3.099547881205403e-01
    4.991970007117742e-01     5.336703013711864e-01
    9.392921094007625e-01     5.308181542175950e-01
    2.609778413136933e-01     9.463283739714867e-02
    3.400021366549472e-01     7.879386155792883e-01
    1.609868281586994e-01     6.537432420196336e-01
    6.661249051569649e-01     6.373994551283466e-01
    9.096583021527227e-01     4.277319929269771e-01
    3.578915849841062e-01     7.412953024722497e-01
    7.406122448819966e-01     7.364661971194630e-01
    4.473264526705921e-01     7.303888234413984e-01
    5.169222571253160e-01     5.638711865260648e-01
    8.368532644764815e-01     3.625674218689081e-01
    6.037670651157264e-01     6.911907709309435e-01
    2.744132077898892e-01     7.461658221576156e-01
    2.686968498794805e-01     7.334989739117548e-01
    1.944863080859393e-01     7.848773233008327e-01
    2.307424613860453e-01     7.443466084953519e-01
    7.808764043704461e-01     5.621900588360553e-01
    9.778420183826371e-01     4.654270175993831e-01
    9.713561463111212e-01     4.613795977638517e-01
    3.286527417752603e-01     6.385375898196288e-01
    9.555884046514679e-01     4.513616261393810e-01
    4.285428451675228e-01     5.455452153875877e-01
    6.055512904654303e-01     9.044067363693873e-01
    3.852614398152964e-01     5.872647478459536e-01
    4.726207127231905e-01     2.292494520779332e-01
    6.305122804882313e-01     5.330774071017033e-01
    4.834174592961866e-01     5.880289010776096e-01
    6.687667964474726e-01     7.537207980129861e-01
    4.063150413010848e-01     4.043562710321406e-01
    7.523637538213700e-01     6.856522977150896e-01
    9.109726722755168e-01     3.359646367159242e-01
    8.894225510140669e-01     6.018859739887037e-01
    5.353131803519499e-01     7.301645377315000e-01
    8.282506187225651e-01     3.716884272571522e-01
    3.350239624836183e-01     4.979295368721567e-01
    1.902376970564176e-01     5.746825870836220e-01
    7.470014758596705e-01     6.566743823206327e-01
    8.597243049637695e-01     5.114608851591156e-01
    6.174634195159184e-01     6.857974685348098e-01
    7.833779392414846e-01     5.034455340688397e-01
    5.775882500534638e-01     6.493716295570630e-01
    4.306274812450963e-01     5.574269366147361e-01
    6.031435500944254e-01     6.208411654770686e-01
    6.311763041395412e-01     3.938966823534651e-01
    7.329166198223174e-01     7.357172799618186e-01
    6.458665477265007e-01     8.649614638871594e-01
    7.408132035682560e-01     5.590579343331777e-01
    7.114385332377187e-01     6.307473937703846e-01
    3.573019151726882e-01     1.794552269895046e-01
    8.777917803475581e-01     3.390957429436262e-01
    3.599230595441522e-01     1.276397298334606e-01
    7.018190639510987e-01     8.500383024007456e-01
    4.086921958749523e-01     2.028382724748452e-01
    3.810850834476576e-01     2.106175570122161e-01
    7.502320314978280e-01     2.597081701332884e-01
    4.399576983824471e-01     3.376837925135702e-01
    7.417093172358270e-01     5.661654887521931e-01
    4.996575482105153e-01     7.086411877078872e-01
    7.973043022786800e-01     4.159064751161405e-01
    3.520856560405820e-01     2.431103475903748e-01
    4.751993810735446e-01     5.333008677201568e-01
    4.911557738211004e-01     6.819011639898580e-01
    5.395903531236491e-01     6.368336073501768e-01
    4.374364068522870e-01     2.566399113934378e-01
    8.542280307617934e-01     2.199379659645629e-01
    3.380994946571155e-01     1.964255473566037e-01
    7.064042229937357e-01     5.811119519390169e-01
    3.605990237890843e-01     4.380602646411235e-01
    3.371165354502716e-01     1.536959903785975e-01
    6.074079370027669e-01     6.705891507451434e-01
    7.436613620331625e-01     6.473174386636088e-01
    2.848072413202217e-01     7.395673690670393e-01
    7.328919677750757e-01     8.701319858354400e-01
    7.772986547444155e-01     2.259713715307747e-01
    5.349576075864718e-01     9.980434841111473e-01
    5.906013180648385e-01     6.037871163934916e-01
    4.867203258814320e-01     9.239332025994323e-01
    3.592454054535396e-01     9.201923748315330e-01
    8.932098165650547e-01     3.023414981643330e-01
    5.691360248491071e-01     7.941211366668149e-01
    4.532635967717629e-01     5.972943858283432e-01
    4.104592566044344e-01     2.332022571302198e-01
    7.214516932253671e-01     5.412311345619786e-01
    8.392079421575176e-01     4.273934846144531e-01
    7.713446595539745e-01     3.020557047377193e-01
    9.212435122978201e-01     5.717110480411629e-01
    4.299981295751952e-01     9.603043893908828e-01
    4.289937757107309e-01     3.680202687222851e-01
    4.793369626037377e-01     3.007149620279708e-01
    3.010566428843726e-01     2.994808323337265e-01
    8.113390103611010e-01     5.504610536587293e-01
    4.047135436047959e-01     3.615773785244540e-01
    7.043730004487030e-01     7.186452221451127e-01
    9.386282411893210e-01     4.238160869271937e-01
    2.643098994992834e-01     1.924848829656050e-01
    4.679093874784829e-01     2.816341148420193e-01
    8.937263946381884e-01     3.083499012974773e-01
    5.780464484653961e-01     8.206113974345064e-01
    6.736697078719615e-01     9.574931522218635e-01
    4.686111491597486e-01     3.864682520603776e-01
    2.232925144269456e-01     7.740959074031032e-01
    4.211654939046148e-01     7.150021746185357e-01
    6.587210148713841e-01     8.183837078910398e-01
    2.265094847019186e-01     7.461289085296909e-01
    2.727897837329982e-01     1.630960277158080e-01
    5.480098618199123e-01     5.073728778200510e-01
    5.333732099427569e-01     5.063977264502789e-01
    4.621046965580762e-01     6.582389332718697e-01
    6.082556854667297e-01     9.620586665318012e-01
    3.198814263957480e-01     3.696539148327539e-01
    4.528424988679750e-01     7.846488168480177e-01
    6.785890571751596e-02     7.432190946583050e-01
    1.030871013883694e-02     7.507479353386828e-01
    5.331132829664819e-01     7.700709641556270e-01
    7.036856230689149e-01     8.436670088056895e-01
    4.702584991378389e-01     6.670035705024985e-01
    8.495847504825929e-01     6.101249583366766e-01
    8.464156597956024e-01     6.298422919102949e-01
    4.920694778433836e-01     5.019486660544082e-01
    6.949016653094896e-01     8.590699832492684e-01
    2.480629825410859e-01     5.440282987319252e-01
    2.650611514735671e-01     7.390547693913927e-01
    2.846924157316385e-01     7.324902701630496e-01
    3.992187883478235e-01     2.635914813526115e-01
    5.980955383683810e-01     4.827085408224956e-01
    2.311239400518470e-01     7.992708688208636e-01
    2.431460150796976e-01     5.388954081613349e-01
    3.833313415313511e-01     9.250290986178267e-01
    6.013157348230993e-01     5.110034064728569e-01
    9.082793128769280e-01     4.173063537286702e-01
    4.035153578878512e-01     3.422943641346083e-01
    5.273250109437806e-01     8.008017907933181e-01
    3.000405160710521e-01     3.774009075093654e-01
    2.947319359857345e-01     6.565059128049215e-01
    3.962681265922352e-01     6.517357200052066e-01
    2.504360315192082e-01     2.487132491322745e-01
    3.552460315400844e-01     1.280183902303443e-01
    4.512302153551903e-01     2.520995353965673e-01
    4.209402448385903e-01     8.270284420575027e-01
    8.240190071846317e-01     6.309004641021747e-01
    4.484395377817444e-01     3.498272075195377e-01
    7.221458298286325e-01     9.281376154652891e-01
    4.950756380983695e-01     6.321084315118768e-01
    4.247430985153339e-01     6.383827733284397e-01
    6.634011861619047e-01     9.565230034494476e-01
    3.838313417868507e-01     6.216293406083307e-01
    5.396425575222851e-01     3.476072220804525e-01
    2.846703021061651e-01     2.145691893808479e-01
    3.631382107713316e-01     3.932776599199944e-01
    8.616692775665834e-01     2.650067757603137e-01
    7.513076379730413e-01     6.574781936697305e-01
    7.875660786147162e-01     2.400835284122770e-01
    3.345829321759667e-01     1.860882010888013e-01
    2.172883121077124e-01     7.065919492257974e-01
    4.836925224853608e-01     5.977170164999878e-01
    2.440408954480040e-01     8.012532572552326e-01
    7.529809485636632e-01     3.045160031601782e-01
    4.554768750985025e-01     7.782882856854161e-01
    4.585547622170665e-01     8.389469517195403e-01
    6.435646690152488e-01     9.678534397007209e-01
    7.513141594697198e-01     3.904378679160210e-01
    3.620898068351556e-01     1.989722585038005e-01
    6.154923969413313e-01     6.640460439940084e-01
    9.244472671704200e-01     5.162781327933451e-01
    4.707388793985368e-01     4.099044688465143e-01
    4.222521984754218e-01     9.386213678441139e-01
    7.854013584404629e-02     7.169782859087160e-01
    8.060088632127397e-01     4.076492230489355e-01
    8.649424324591092e-01     4.609416269558506e-01
    2.856848657621972e-01     6.766821339796378e-01
    4.857135447245925e-01     2.636838894525312e-01
    6.305786721892124e-01     6.983476473144553e-01
    3.447971414100706e-01     3.130302402928894e-01
    7.798442050047331e-01     4.225622958595963e-01
    3.895517711017388e-01     1.886817298405228e-01
    5.310670595463350e-01     6.191927637223312e-01
    7.881252252634225e-01     4.077251786136309e-01
    4.884199717028641e-01     4.320161651445792e-01
    8.654350522144157e-01     4.437215453880408e-01
    4.648363723631432e-01     4.953341581618264e-01
    4.005134028117869e-01     8.136668653657235e-01
    2.475058078490311e-01     8.423110230949634e-01
    5.450193721632987e-01     6.369814876894664e-01
    4.082714638843785e-01     7.446180200954597e-01
    6.716486713109169e-01     7.329963958792877e-01
    2.551077061572258e-01     2.031576423629292e-01
    8.863026412697921e-01     3.876883681399196e-01
    4.756864457517836e-01     7.410624349634761e-01
    7.948507078525752e-01     1.978647848880944e-01
    6.828389608455149e-01     8.528305023265457e-01
    2.914343588152508e-01     7.617791433936953e-01
    4.399157836372367e-01     2.121655887232300e-01
    4.375714994387903e-01     2.091543863990789e-01
    3.160555744747818e-01     2.785895042160379e-01
    2.808492690996374e-01     7.751088605944897e-01
    7.088492984204836e-01     5.825120547913337e-01
    3.735732637036511e-01     2.417635277681580e-01
    8.021455198892646e-02     7.366053050913242e-01
    2.870198840354165e-01     3.910929343599250e-01
    2.890817437789941e-01     8.636247158180025e-01
    4.944156507579278e-01     4.230950861784650e-01
    2.864008221619825e-01     8.792221408114569e-01
    7.729045724068638e-01     6.000049369833648e-01
    4.138222273935426e-01     4.124989523570359e-01
    3.027808463404292e-01     5.205036259044336e-01
    2.840451501678376e-01     1.616447863189850e-01
    6.523986442590111e-01     8.732299514372736e-01
    7.184151368576597e-01     7.444221909193863e-01
    9.095662405300373e-01     5.397206027354090e-01
    3.061746945153431e-01     5.048300000058129e-01
    7.404072126749734e-01     9.388404151309324e-01
    4.687369197070130e-01     4.789940962232955e-01
    6.327729306208931e-01     8.597437105498175e-01
    3.996038543754654e-01     4.714214239027261e-01
    8.413866490632572e-01     2.068557385902736e-01
    4.571966604546245e-01     8.184806109489599e-01
    6.525391719587608e-01     9.942593921122690e-01
    6.929582769527927e-01     8.328209451234769e-01
    4.276088413450051e-01     2.246593748320084e-01
    7.583641556460964e-01     3.642327170165607e-01
    8.057220902814154e-01     4.282425790109546e-01
    7.434283835243264e-01     9.916247567974449e-01
    2.599489860698929e-01     2.910239080977098e-01
    6.872130898683247e-01     7.858809708851232e-01
    7.832273079930561e-01     1.782774534240333e-01
    7.203191737798087e-01     8.760224387477278e-01
    2.761027306634225e-01     8.005973482470382e-01
    6.360883254294791e-01     8.864215693172705e-01
    7.373248270631249e-01     6.079875359862786e-01
    8.092793173433399e-01     3.262714602983173e-01
    1.624943813315596e-01     6.517383078946064e-01
    4.581210549224598e-01     2.732913048945075e-01
    3.586120699286149e-01     8.737519490214208e-01
    4.004491791338834e-01     4.194799547711017e-01
    2.474636010162157e-01     5.221400957492587e-01
    1.855978104179630e-01     6.659736636120895e-01
    7.505428180308131e-01     5.430570032256135e-01
    4.406390446887626e-01     3.177357990433340e-01
    2.332657721450024e-01     7.359079073520466e-01
    5.014775012602818e-01     9.639703035960594e-01
    3.601659082087429e-01     2.185714189021375e-01
    3.785473054178095e-01     3.000835765256011e-01
    3.229248929383749e-01     2.778912026329898e-01
    4.495044795462900e-01     9.692776857846824e-01
    8.085792768432315e-01     4.009139531352840e-01
    3.700148184093088e-01     3.482682888181459e-01
    3.150805066606293e-01     6.821384476112948e-01
    5.477616949824099e-01     6.637691908531177e-01
    5.649646908400182e-01     9.985930541586947e-01
    7.036419499174466e-01     4.778178307522913e-01
    7.786249402973584e-01     4.530023089945346e-01
    7.233490401886469e-01     8.443974468742834e-01
    7.998847608065809e-01     6.274570054022519e-01
    2.275341599254092e-01     8.306863738610222e-01
    2.891706603252834e-01     6.765096953326796e-01
    4.347444519840250e-01     2.364276955636176e-01
    9.282837290233853e-01     4.256215688328721e-01
    5.664532902645463e-01     3.558344379145731e-01
    4.460464348526654e-01     3.515247593737150e-01
    4.686707587755625e-01     2.910305923053493e-01
    3.057379776900208e-01     8.201348781159005e-01
    5.857460600626789e-01     9.921856044295905e-01
    2.782541918727031e-01     7.165845484991148e-01
    1.746789428088020e-01     6.895149141270597e-01
    4.977876716638455e-01     3.717711737925538e-01
    3.886659596387382e-01     3.937541998464084e-01
    8.708740212397844e-01     5.276658957868823e-01
    4.868700943709483e-01     9.378330931389470e-01
    9.221724878798867e-01     4.944352252713875e-01
    6.236215285216985e-01     5.876648536908123e-01
    3.422921094543337e-01     8.404219347174978e-01
    6.665885342416367e-01     6.951436046093264e-01
    5.352473619607193e-01     8.178373396606806e-01
    7.505463170736111e-01     3.143346949519303e-01
    4.255487651474074e-01     6.292563933584695e-01
    2.910072277059578e-01     2.394451960333491e-01
    2.811354325647110e-01     1.069701874151745e-01
    4.201694835693620e-01     1.998451510584984e-01
    1.326186038884876e-01     6.844333640267994e-01
    6.998209155558358e-01     6.792419374879879e-01
    6.219603157218500e-01     4.085689608556613e-01
    2.916688463284905e-01     5.052185187881699e-01
    7.540569483012048e-01     5.123704252573761e-01
    3.848051599839999e-01     3.701761223901814e-01
    9.358386591766273e-01     3.780353509557765e-01
    3.881701685309462e-01     8.216473397850936e-01
    7.362604991052883e-01     5.821596173320429e-01
    9.220521568873224e-01     4.265688137034882e-01
    5.601195611785054e-01     8.924258986827437e-01
    1.158896443538490e-01     7.801341039426343e-01
    4.456242646702593e-01     5.606243144606933e-01
    9.050737305638012e-01     4.204706486451966e-01
    4.749079043386720e-01     5.887369083799493e-01
    3.686211306479877e-01     2.628584084710268e-01
    4.150275046280243e-01     9.526202069574291e-01
    3.546807588098738e-01     2.193281051894653e-01
    9.802319940187425e-01     4.811245714588481e-01
    6.160565101120604e-01     9.223268579782593e-01
    4.132632319832644e-01     6.910476063695147e-01
    3.864819975394773e-01     6.628111797390879e-01
    7.289247283731928e-01     9.365571883901833e-01
    3.142417559230036e-01     2.763046845134991e-01
    6.241228205524314e-01     6.608148175667831e-01
    2.858952407792686e-01     2.409685004211957e-01
    3.915732654644444e-01     7.356017753007223e-01
    2.080908250986860e-01     7.018057208209533e-01
    7.199046997211658e-01     6.369827643310235e-01
    4.564075063399093e-01     2.225271738225493e-01
    1.646528720546052e-01     7.027774417545060e-01
    5.942882065025464e-01     5.017096440584513e-01
    2.535079504458599e-01     3.540821141719226e-01
    1.879780754854364e-01     6.729826571612495e-01
    3.574716278649566e-01     4.891803045470161e-01
    1.688564589527228e-01     6.090329886787013e-01
    4.322218641766817e-01     7.079625003388442e-01
    5.086979206539777e-01     2.631040014005683e-01
    2.363695159243174e-01     5.367486343181855e-01
    5.122507311078255e-01     7.975792988812352e-01
    3.338351154161597e-01     8.840254685639803e-02
    5.263234568976920e-01     3.985241569958845e-01
    7.525294725091271e-01     6.384522569993751e-01
    3.042124049888032e-01     6.580561248820925e-01
    3.141007150276635e-01     7.026028776313782e-01
    2.968565259312332e-01     5.803360374366507e-01
    3.138419472750128e-01     8.838269063430771e-01
    8.896043404261218e-01     3.927077375975750e-01
    3.977397346500083e-01     3.087784761194738e-01
    1.989316086527664e-01     6.068963684409842e-01
    4.491404541594503e-01     6.446038747743489e-01
    6.011474824192375e-01     9.194456158021231e-01
    1.894515381851505e-01     6.622549695658657e-01
    4.784040803152331e-01     7.083097340219604e-01
    1.869731078669702e-01     6.951408953004867e-01
    6.739297810604905e-01     5.932895791597326e-01
    6.855545209119920e-01     9.262389470691071e-01
    8.360019428957233e-01     4.238802101637905e-01
    3.489595756573309e-01     3.107226543995718e-01
    2.168596280848821e-01     6.629678353414808e-01
    2.776700384198767e-01     6.653099426136782e-01
    4.298234208327942e-01     5.553303253394589e-01
    4.345737761655306e-01     5.913672914251232e-01
    3.128749434627833e-01     1.500112383354456e-01
    3.049589694930545e-01     2.797290111408768e-01
    4.569235306689762e-01     7.734410424802919e-01
    7.136039587870590e-01     9.930860631498981e-01
    7.292760139438399e-01     7.041111677603415e-01
    3.041934992639835e-01     1.652811568955067e-01
    5.402249386582465e-01     7.333499489015832e-01
    3.191742967703636e-01     2.895379796938822e-01
    7.467332159152644e-01     7.608219938297117e-01
    5.343691460420760e-01     3.975492080481766e-01
    5.649938757333063e-01     5.217007910024473e-01
    3.627773558025105e-01     9.070278954860118e-01
    2.698996633161698e-01     3.555845806434608e-01
    7.126757419313116e-01     6.629697785656083e-01
    3.531993627654003e-01     1.157970085138573e-01
    6.911033090540132e-01     5.979675818434443e-01
    2.584619511696855e-01     2.687009412876699e-01
    4.869775870992192e-01     5.796718562237891e-01
    5.490479716751754e-01     7.458888944840311e-01
    2.521566629739578e-01     3.527490692235988e-01
    7.308458545204293e-01     6.357020801953698e-01
    3.742015407242257e-01     5.487595464532531e-01
    6.203828668793266e-01     3.920214420014143e-01
    6.862940880944263e-01     4.414601847703123e-01
    7.816046557131077e-01     1.820521306324093e-01
    8.600394550577422e-01     3.975623145698783e-01
    4.736439760769164e-01     7.377090401026726e-01
    4.843055145388474e-01     8.707696949515896e-01
    7.903882000190066e-01     9.635422800781712e-02
    2.848868545065997e-02     7.271831332895076e-01
    7.565325616276036e-01     5.745792492805345e-01
    4.246407263263859e-01     6.457125421467224e-01
    7.644931177409877e-01     4.542515988692006e-01
    5.415177962960650e-01     4.697408695156674e-01
    1.634090409000510e-01     6.124986387185144e-01
    1.027794996513665e-01     7.613074905878319e-01
    5.355242027879330e-01     3.557705608345073e-01
    3.460092409735529e-01     1.699146916908141e-01
    2.708958270748536e-01     6.545616982346960e-01
    3.065699063162658e-01     4.558313688932847e-01
    7.255358051750823e-01     6.471098277888299e-01
    2.127686617614853e-01     6.310574613758554e-01
    4.412227412165310e-01     6.846039495642054e-01
    3.866634398305724e-01     7.843856933577074e-01
    7.764361026996881e-01     2.171603731646244e-01
    5.740677501367337e-01     5.820180836908992e-01
    6.047103741293824e-01     7.708114716578923e-01
    3.782879297426168e-01     6.865332724723311e-01
    3.554048320648237e-01     8.186732593043724e-01
    6.573133724308390e-01     5.187156246084431e-01
    5.791293824608599e-01     5.029881300573491e-01
    6.650072020841085e-01     7.666515784335602e-01
    5.777114314360661e-01     9.825319270063541e-01
    5.060034631657087e-01     5.241009977007095e-01
    7.654599215417921e-01     4.760480824520786e-01
    8.365933320579578e-01     3.326520808984659e-01
    3.077081452547962e-01     3.906154390242850e-01
    5.129207208604510e-01     5.329857754442712e-01
    3.323512051885111e-01     5.959943370413147e-01
    7.710726094417523e-01     4.976818610380166e-01
    2.652752200534394e-01     8.069804712485772e-01
    5.996993128928768e-01     3.566460245412421e-01
    6.268369377587005e-01     7.333269895135855e-01
    5.498970362319050e-01     6.514937729760468e-01
    5.473786574821445e-01     4.875550404583791e-01
    3.036384446207097e-01     8.084440694227452e-01
    6.407128688550545e-01     6.092534082654969e-01
    5.430527692968171e-01     5.787068756102361e-01
    7.175189872849609e-01     4.992634813238365e-01
    5.355182398010023e-01     5.394238800978898e-01
    4.311310238654626e-01     4.030578343232954e-01
    7.333660673146237e-01     7.990976224113453e-01
    2.960886703077734e-01     7.601226723633533e-01
    2.563522984162161e-01     1.122672938771863e-01
    9.588073203191441e-01     4.331084525777352e-01
    7.774119378884217e-01     7.138567678875178e-01
    2.653584034096010e-01     2.642332118251516e-01
    4.160162782933088e-01     9.432830003859486e-01
    5.463985964821917e-01     7.466748624455750e-01
    6.494441923892271e-01     9.087214265441322e-01
    8.367381161824115e-01     3.248491424836472e-01
    8.819189131212942e-01     3.532886557899924e-01
    4.132059388990903e-01     5.632752878727156e-01
    7.182535657359913e-01     7.214557946581694e-01
    7.238111207397863e-01     4.966033294950229e-01
    8.257013240375078e-01     2.549711521799166e-01
    8.711159959532317e-01     3.166224505570888e-01
    6.695194148434287e-01     7.620047514203666e-01
    4.940971091192718e-01     7.876005198282655e-01
    4.125101072578909e-01     3.718713079865292e-01
    5.330423570827278e-01     6.992150685201513e-01
    6.215448108435173e-01     7.120060435909961e-01
    6.788731613208031e-01     5.397487479420042e-01
    5.660330315157501e-01     7.155550915349336e-01
    5.804380396088905e-01     4.738636288712296e-01
    4.475588157263487e-01     8.908980295570266e-01
    2.722667217938770e-01     6.163543232387766e-01
    3.961614920034415e-01     3.149473588169749e-01
    6.038989887562645e-01     6.888310995445565e-01
    8.123856107279183e-01     1.538897636142764e-01
    4.876984537936453e-01     5.940189177272559e-01
    6.009508361383344e-01     5.131083484992042e-01
    8.417065468525521e-01     3.625430587428703e-01
    5.230158537838927e-01     8.483927798877705e-01
    2.618655555925836e-01     2.282781978318628e-01
    3.468163252411882e-01     6.894310126397996e-01
    8.600546527881805e-01     5.667860516613149e-01
    4.076617083429981e-01     3.233070650868153e-01
    2.540306113755497e-01     5.292267260875081e-01
    2.995060681771867e-01     2.873565802937643e-01
    7.130593686650958e-01     4.950476715156600e-01
    7.182921060446767e-01     5.263592985629554e-01
    4.583249513332337e-01     8.542930351547665e-01
    8.375424866882621e-01     5.291305217952204e-01
    4.870676758880613e-01     5.451715616299115e-01
    2.330231840369794e-01     7.418513950383403e-01
    3.731727779766570e-01     7.428952856234192e-01
    3.678657804325953e-01     8.984935627680516e-01
    5.679410606381166e-01     8.188177406897579e-01
    7.063105993757381e-01     6.320104926972573e-01
    8.814958125199976e-01     3.568732116262820e-01
    5.292033484894862e-01     6.830668583828732e-01
    4.411363635207536e-01     5.208595496162854e-01
    4.510628798202244e-01     8.943020091271240e-01
    7.666403649556939e-01     7.332135906831551e-01
    7.175597860370095e-01     8.849046670367599e-01
    6.128698923996723e-02     7.030522335255395e-01
    2.759765375442279e-01     3.426191058667930e-01
    4.267447560437307e-01     9.585050990743503e-01
    6.075578107760398e-01     8.436833362144143e-01
    5.834992114265279e-01     5.544140476720134e-01
    6.535841913473849e-01     8.895588085775537e-01
    3.949932953690805e-01     8.151644172322091e-01
    7.638483684337468e-01     4.707456439480829e-01
    7.983560988324538e-01     2.894029034891220e-01
    3.745889295700296e-01     6.339033324085631e-01
    5.332504182416234e-01     3.774492129695302e-01
    4.559006461023387e-01     7.368769981390050e-01
    7.125183129160101e-01     6.429927363710028e-01
    6.416197683196898e-01     7.165598152289999e-01
    2.690111547903953e-01     5.187361190014863e-01
    7.586342178926052e-01     2.226947661255703e-01
    5.001343953613837e-01     4.105886716109570e-01
    9.132688145078409e-01     5.204297502037092e-01
    7.839600013133488e-01     6.596092106968602e-01
    7.061261987348065e-01     9.519266260653371e-01
    6.634609490010125e-01     4.279580198866397e-01
    4.261944708896421e-01     2.129619829522229e-01
    3.971107269093588e-01     7.255940739318176e-01
    3.372962861624549e-01     2.182508852578066e-01
    9.005351742496270e-02     7.088340091577650e-01
    8.241982556049835e-01     4.005973559400775e-01
    7.476775746215694e-01     9.142394941571254e-01
    9.234802445131995e-02     7.875941112153215e-01
    7.694190453492353e-01     6.245410361596927e-01
    3.297580654934432e-01     4.978967585115915e-01
    4.585140739508420e-01     6.364133864028120e-01
    3.457482476923303e-01     5.461993921727029e-01
    6.294302428316757e-01     8.862629175097416e-01
    2.368452457956894e-01     5.941840243163361e-01
    6.553852296985997e-01     8.811094833220664e-01
    4.628226704194057e-01     8.110391894838876e-01
    6.137593485293735e-01     8.189067575647843e-01
    2.917954780917102e-01     5.252118312829930e-01
    6.120936260513454e-01     4.023489002158930e-01
    6.804795866082327e-01     9.848912552083698e-01
    2.005805980351569e-01     7.905869407995826e-01
    1.659602329449668e-01     7.003780305700645e-01
    7.454311318422083e-01     7.337613024704646e-01
    3.751813825270662e-01     6.900975709288197e-01
    5.142367645163844e-01     8.033922932981947e-01
    6.995694522180047e-01     6.618114727683914e-01
    3.811938293318987e-01     6.242192469787861e-01
    4.766493526815750e-01     8.754888352120550e-01
    2.721636852199243e-01     5.741720664288735e-01
    1.281696284903517e-01     7.050509734321568e-01
    6.976208007923805e-01     5.437691973194358e-01
    7.691248605878801e-01     5.218979405496250e-01
    3.126584389899695e-01     4.810666534405229e-01
    4.733625647722955e-01     3.731740702344952e-01
    6.508951783386304e-01     6.526547883055878e-01
    6.271469840881535e-01     6.662871999370054e-01
    6.249970660348922e-01     9.954583387427690e-01
    2.922137652260919e-01     8.114472800132704e-01
    6.397717897434664e-01     5.908421006351662e-01
    4.400665293117959e-01     5.660514955318390e-01
    3.212549445512783e-01     8.326142555563376e-02
    8.070993712131067e-01     5.008255925625953e-01
    8.662293701165316e-01     3.336097306002057e-01
    3.913597105942558e-01     1.520324472038622e-01
    8.079782936618657e-01     1.657385805946170e-01
    5.164063815158324e-01     8.811085886223596e-01
    6.634810759526829e-01     4.733352234585739e-01
    5.027060083633967e-01     4.956259107441467e-01
    7.874402309555293e-01     1.317062362011394e-01
    7.727817336641786e-01     4.364827592377934e-01
    5.892088373323827e-01     4.121942751037853e-01
    6.859163314621529e-01     9.817190988153217e-01
    8.049995293993099e-01     5.690001970192142e-01
    8.499646781078105e-01     5.206280375846771e-01
    2.256237709598190e-01     7.691546923488027e-01
    4.251119378514787e-01     7.084609453146171e-01
    2.854986508817053e-01     6.998593435965951e-01
    5.511909009051268e-01     6.007494465207396e-01
    7.772647099051473e-01     7.619914066427791e-02
    5.134674563113926e-01     8.186851542420872e-01
    6.236049336985104e-01     6.383469178616710e-01
    8.082917959299530e-01     6.318086964396429e-01
    1.918164773605471e-01     6.804362141498853e-01
    2.906017759237696e-01     7.371593287949141e-02
    6.112624999590149e-01     4.041949930492686e-01
    2.987600045515434e-01     5.023508832210366e-01
    8.890907690274632e-01     4.705297857929520e-01
    4.129152939379180e-01     7.922538566677289e-01
    3.266713775160111e-01     2.363410012718355e-01
    6.621339276296857e-01     5.469449135874714e-01
    8.806059783616093e-01     5.066436110950930e-01
    9.479953512150966e-01     4.761856083487502e-01
    4.329829789746626e-01     6.035974942926626e-01
    4.427139356379918e-01     8.783173986843179e-01
    5.951308145873274e-01     5.116970774088139e-01
    5.148334729652971e-01     6.304372293347706e-01
    5.580661849529065e-01     7.138679166127638e-01
    5.544118396797320e-01     3.213323200859879e-01
    7.548992338894464e-01     1.310399560804343e-02
    8.141440456007801e-01     5.716677931377155e-01
    5.803166909841350e-01     6.817331779864633e-01
    8.689870099533632e-01     2.804074379519174e-01
    4.065772725575915e-01     2.109833950987079e-01
    5.985009917200862e-01     7.619444965396005e-01
    2.761634438443965e-01     1.728084155283922e-01
    7.261094595992332e-01     7.479178635182031e-01
    2.644640126095673e-01     5.100196483598715e-01
    4.162974847056337e-01     5.523445011630280e-01
    3.014175197892685e-01     7.100427476435694e-01
    3.735026476296855e-01     8.041065535750384e-01
    8.059373135756668e-01     5.862076550066970e-01
    8.958369320145414e-01     5.671537354156599e-01
    4.488394011853031e-01     3.227124686814764e-01
    4.168254543035368e-01     6.245005201846388e-01
    1.607523927330894e-01     6.086673750101984e-01
    4.427387396140399e-01     2.164826550747054e-01
    7.660753026498850e-01     7.292243812761641e-01
    7.727119432579532e-01     3.457213802803866e-01
    2.472187492722786e-01     6.616417706523876e-01
    6.458057361821993e-01     4.479340621063383e-01
    8.837723854386564e-01     5.444951483319049e-01
    3.140921084907889e-01     4.475424792265085e-01
    6.322430165260250e-01     7.919154952429683e-01
    4.831473052341658e-01     5.627815870209004e-01
    5.474257965071622e-01     6.068573445867227e-01
    2.336573685946380e-01     5.174368107654020e-01
    3.571175372552415e-02     7.501448449580644e-01
    2.802774851355357e-01     8.502485068024697e-01
    7.794772632259480e-01     1.260663966241199e-01
    6.160690209993404e-01     5.256032411680189e-01
    7.498928528247520e-01     6.163596049378773e-01
    3.417542012930510e-01     2.317563712595182e-01
    6.825853934715012e-01     8.636045841595299e-01
    5.144442139700761e-01     9.342345624895978e-01
    8.331852952749783e-01     3.334329869938875e-01
    4.408189529327293e-01     6.736252199271017e-01
    6.746740927394261e-01     6.404657903519511e-01
    6.211273332946821e-01     7.867234665619529e-01
    9.107927017475237e-01     4.046194590028321e-01
    4.250754271723011e-01     7.565305918425830e-01
    9.264127340154071e-01     5.009575175702206e-01
    3.881871481976217e-01     5.168941291464989e-01
    6.217254730658484e-01     5.119295220488698e-01
    5.710132306958833e-01     4.368182080394770e-01
    4.027589268425982e-01     3.427198922096016e-01
    1.058522744861222e-01     7.234776756745277e-01
    5.953215564462649e-01     6.898838932599437e-01
    3.242539705145199e-01     8.907956542425459e-01
    3.048323059662220e-01     3.247649247230128e-01
    4.671569604358029e-01     8.260620007960303e-01
    2.085251111911056e-01     7.734826874001798e-01
    3.324199748121043e-01     8.772603492642630e-01
    4.114949919791748e-01     6.474961262531389e-01
    3.339356434293674e-01     2.347744511139526e-01
    2.027074711658990e-01     5.971505142428081e-01
    6.064900948760616e-01     7.077912975482485e-01
    4.611038484170356e-01     5.831300496574390e-01
    4.795285077341122e-01     8.894972278061206e-01
    6.114291555227201e-01     8.051837992917394e-01
    7.722717006270527e-01     1.929186987608328e-01
    9.159694328778593e-01     5.177070452840237e-01
    3.590036113638825e-01     3.247805938992083e-01
    6.755989854771889e-01     8.117315611464025e-01
    5.615930522682085e-01     8.210710106283113e-01
    1.740411739155803e-01     6.620172390894595e-01
    6.498974275745765e-01     4.730991043840330e-01
    7.386931558476810e-01     8.870366203612027e-01
    2.939955369722418e-01     6.922630515698266e-01
    4.525792813690310e-02     7.114302424048140e-01
    7.911334715611370e-01     4.573569141378812e-01
    6.665706951414030e-01     4.279841059586413e-01
    3.806827272891251e-01     6.327994385884037e-01
    8.037840590902621e-01     6.380585535196289e-01
    2.905396823346102e-01     6.276241663959586e-01
    6.370228344134260e-01     9.591444979683117e-01
    6.685630686750627e-01     5.137907560285920e-01
    6.366318189000998e-01     9.680443916699165e-01
    2.784979400540510e-02     7.441925600266136e-01
    6.578664777938330e-01     5.148129540290209e-01
    5.382488712084936e-01     3.541405237196036e-01
    5.040656617815643e-01     2.559502522565803e-01
    3.750473107318123e-01     1.379068593116531e-01
    3.415121206349866e-01     3.216336713432599e-01
    7.396106829050666e-01     7.557705767317364e-01
    6.369495697543227e-01     6.509603913187377e-01
    6.170293699353154e-01     5.803066723761136e-01
    2.726515158897003e-01     8.767515552943919e-01
    6.319446801847697e-01     7.291284620415202e-01
    4.708041764092151e-01     4.261342275704111e-01
    4.162896961977538e-01     3.367552759401897e-01
    7.733491994071386e-01     4.820690756405661e-01
    8.909339418164750e-01     5.177089784328737e-01
    8.136404568347673e-01     4.300482669871912e-01
    5.649908773652873e-01     7.888553603078359e-01
    2.582175189790844e-01     5.007729988517267e-01
    4.274898700836167e-01     7.600601741152366e-01
    5.084525931003531e-01     6.696467828343270e-01
    2.963323577148780e-01     8.503997634730565e-02
    6.040828511736729e-01     4.906764953632381e-01
    7.513998480084193e-01     7.078111069642887e-01
    7.486868747376406e-01     9.552796387881644e-01
    2.539626496301846e-01     4.024393413690948e-01
    5.798656039514948e-01     6.589610809042304e-01
    3.764333025777176e-01     4.895158205848222e-01
    6.523435640551744e-01     6.784410547826065e-01
    6.042789472421990e-01     5.713942670572263e-01
    6.933523520087266e-01     7.660807745386943e-01
    4.709637995964132e-01     8.754684768523057e-01
    8.366138321666921e-01     3.464463609046431e-01
    7.212136348252618e-01     9.073654706649913e-01
    4.960018204689394e-01     4.993625368916077e-01
    4.292667498490880e-01     5.247488014146717e-01
    3.386996269764866e-01     2.758959533266959e-01
    8.765378414110466e-01     2.904771909659647e-01
    5.014405364912626e-01     6.760595989876942e-01
    4.374783142539038e-01     3.767578123730896e-01
    7.601221821255698e-01     4.333254940540935e-01
    3.945437904991569e-01     8.733428535866866e-01
    6.056212631382110e-01     4.733630690558072e-01
    4.433542814111059e-01     7.731730805857471e-01
    4.988747254502773e-01     5.194323752465075e-01
    2.696659108752218e-01     6.374444358027362e-01
    6.801901985588585e-01     6.874122935763457e-01
    6.418721992133989e-01     7.584712813204088e-01
    6.262664268064544e-01     7.876727195216087e-01
    5.522057650120293e-01     4.223794121235143e-01
    5.602372355384178e-01     8.405629745960478e-01
    3.150884355339658e-01     7.469844354730635e-01
    2.166960932319264e-01     5.868534951285705e-01
    6.222883016574652e-01     9.350980815987503e-01
    3.891841179938791e-01     6.953558886802201e-01
    4.774891049089695e-01     3.833817000614317e-01
    6.011516584343422e-01     4.560932010790175e-01
    1.069516646167363e-01     6.657563293546890e-01
    5.249031458574001e-01     5.662330024032908e-01
    9.541575898607431e-01     5.239044134816704e-01
    8.934918527752113e-01     5.503089244275222e-01
    6.960161945627831e-01     7.356168906678658e-01
    3.282540995310110e-01     8.444251096226673e-01
    5.556683513742963e-01     9.733209611498376e-01
    2.606531527780882e-01     3.447476907437018e-01
    4.864510833715594e-01     4.676842158297283e-01
    6.036101821934342e-01     4.493315232961100e-01
    3.115019873720182e-01     5.402609000880592e-01
    5.341996695009692e-01     9.539544397169656e-01
    2.701486673245109e-01     3.143459789052971e-02
    3.355339090772117e-01     5.306923037216769e-01
    3.440797406614670e-01     5.896027252858385e-01
    5.928465592032970e-01     5.715212373815934e-01
    7.514474115674779e-01     1.283241997524900e-01
    7.003354621240397e-01     8.139459405017387e-01
    5.614400345001678e-01     8.955229296538575e-01
    8.788445325969084e-01     5.550297253915545e-01
    5.852529040757461e-01     8.772741125575789e-01
    4.355463609284277e-01     2.923916543724139e-01
    3.621554194982137e-01     3.323995439539621e-01
    6.552646622334696e-01     7.136360594782378e-01
    2.916735390405627e-01     4.256014160501064e-01
    3.383708744431387e-01     2.051027337050699e-01
    7.429980536684780e-01     5.588307135172743e-01
    6.456635109417797e-01     4.871256324229214e-01
    7.505453090472038e-01     3.529215083221023e-01
    8.203846252371487e-01     2.414460032333933e-01
    2.606884845142133e-01     5.468498093412193e-01
    2.926818256464951e-01     3.496874641539490e-01
    8.250832763068110e-01     2.980079094839515e-01
    7.755195657453421e-01     3.912390032792489e-01
    2.041198086768824e-01     8.139404439463976e-01
    2.875007487494692e-01     4.233995606252872e-01
    3.009841943769594e-01     1.443804993819028e-01
    9.048591065508571e-01     4.005900053279488e-01
    6.309288317959320e-01     8.231999259005415e-01
    6.033132629914716e-01     6.464402285700040e-01
    4.892194978760064e-01     3.819675484768764e-01
    5.433403257968078e-01     8.354178533952854e-01
    4.724370515719049e-01     3.832750805670969e-01
    7.581136404996701e-01     6.367029487050491e-02
    8.400696864492947e-01     4.838065841170217e-01
    6.721185879689225e-01     8.629012959153629e-01
    8.461992470415715e-01     6.143272611207283e-01
    7.512199617738365e-01     5.716415309159369e-01
    8.746789423561514e-01     3.645530250439020e-01
    1.230035889176522e-01     6.660041302128268e-01
    2.880627641417526e-01     4.069148723856539e-01
    6.218415329843953e-01     9.335087382255179e-01
    4.380048357147747e-01     7.802664615532877e-01
    4.022887559209639e-01     1.848273632224141e-01
    5.306286909688772e-01     9.550179791692823e-01
    2.970572157219445e-01     3.608056852727757e-01
    4.359295545652255e-01     4.930093328064813e-01
    1.985702343001543e-01     7.822171480258512e-01
    3.181687946262821e-01     6.433702314214498e-01
    9.159252788642602e-03     7.491254591677428e-01
    8.435623928548697e-01     4.527032205930520e-01
    4.885508427671931e-01     3.441872217121817e-01
    3.617511506940982e-01     5.833544761044618e-01
    4.547691716182594e-01     9.670104735959355e-01
    2.543431324894301e-01     1.865008653132281e-01
    8.453038509792609e-01     4.397667676790662e-01
    2.726273405004163e-01     8.499979295030424e-01
    8.067855306291132e-01     4.033502336713169e-01
    4.699571525178704e-01     6.779559932596221e-01
    6.147913622735212e-01     7.259499323547899e-01
    4.034008319360327e-01     6.960367544382934e-01
    3.048548626673413e-01     5.153691066622897e-01
    8.119323294097737e-01     3.192773294150864e-01
    2.784191336593188e-01     8.305257964733579e-01
    3.096552102370493e-01     1.777148830142642e-01
    4.493876780798546e-01     5.132299412254100e-01
    7.098258367996667e-01     7.652844352941505e-01
    6.589047045249998e-01     5.441041337025903e-01
    4.143246409219487e-01     7.543699491316428e-01
    3.613634959558621e-01     4.711518747890275e-01
    2.780908305748172e-01     3.470937739376611e-01
    5.891385549609263e-01     8.706901671151760e-01
    5.748782006413030e-01     6.872279278432148e-01
    4.396119965576726e-01     8.616073872341494e-01
    5.825291019129469e-01     4.102128246206922e-01
    3.410327486629345e-01     4.299121809649461e-01
    6.174246650919654e-01     5.866816843320686e-01
    5.686187785891367e-01     9.717973751590159e-01
    9.204433102787908e-01     4.422168665313195e-01
    8.183725650186068e-01     2.047916663569174e-01
    6.364419339844796e-02     7.573616775399217e-01
    6.891886914775841e-01     4.996695507797105e-01
    7.645351235274855e-01     2.131518248944807e-01
    7.620676923594363e-01     6.497195462442293e-01
    5.796191461908149e-01     4.265568522293095e-01
    1.818846233703892e-01     6.152810806299893e-01
    7.749300377332241e-01     6.256246889356710e-01
    5.113255276633388e-02     7.753391072137362e-01
    3.581722250260364e-01     7.813673695116938e-01
    5.187117966885029e-01     7.787713134795867e-01
    5.081383746877004e-01     4.702169729368004e-01
    6.915902118151899e-01     4.763881126298022e-01
    2.140025111584831e-01     6.657356498493692e-01
    2.796184884873435e-01     5.861682979170801e-02
    1.622731404268588e-01     6.173186689645246e-01
    4.400875319784997e-01     9.062393969549509e-01
    4.990479193507320e-01     6.019430074987139e-01
    2.663103564449852e-01     5.631187817988389e-01
    6.247987964729685e-01     4.318828460185736e-01
    2.863808892613217e-01     2.055359427976365e-01
    3.426668800331057e-01     3.640772960027541e-01
    8.599500281071498e-01     2.321598088854394e-01
    7.972001474048886e-01     5.738876601609674e-01
    3.561123021641266e-01     3.830796658553087e-01
    6.921015375674163e-01     8.767299075245920e-01
    4.708990761741519e-01     4.183472485817994e-01
    7.145985340777868e-01     5.912573908513244e-01
    6.306078789555701e-01     5.997334640429206e-01
    7.453122113459082e-01     5.715187169063098e-01
    3.969586950737322e-01     8.540592703476395e-01
    6.030098544110688e-01     6.008736420045322e-01
    6.045620248002689e-01     8.909484659309549e-01
    5.306325789076574e-01     8.117802207254069e-01
    7.361930042894720e-01     6.733801369754308e-01
    2.792835978597111e-01     6.677788942866356e-01
    4.338894978341810e-01     8.275538931735424e-01
    7.744422849927300e-01     9.872773774696175e-02
    6.241255796428095e-01     9.274135594994227e-01
    4.096709125532664e-01     5.267325754144957e-01
    4.879799017101325e-01     2.391342307209547e-01
    5.204193759149137e-01     2.970927055353136e-01
    6.208176631170588e-01     5.227765449111789e-01
    8.805165129486174e-01     5.973242087869624e-01
    3.624616300758368e-01     8.389667022147537e-01
    7.564571044847793e-01     6.754524670579176e-02
    2.612880742696251e-01     6.817733238112512e-01
    6.696081741618382e-01     7.751186421168856e-01
    4.456032198855912e-01     9.401192149276909e-01
    7.082122325465324e-01     9.579796454719399e-01
    6.292759322842023e-01     8.601500182164579e-01
    6.415315582238232e-01     7.781868483421418e-01
    5.264537866543197e-01     9.473716019070266e-01
    8.408757069629081e-01     2.476978400562865e-01
    6.079492714118173e-02     7.529480804235983e-01
    4.265625720338974e-01     7.781024875027983e-01
    3.407050441420403e-01     4.977745509034572e-01
    5.989807796515662e-01     9.754840389296879e-01
    2.171425322475308e-01     5.571813062264830e-01
    4.321366205046822e-01     4.079237654695069e-01
    3.495872879738364e-01     1.148324406986176e-01
    7.905037186415347e-01     4.290530226500101e-01
    5.698346225372809e-01     9.641175330795454e-01
    5.687270029615684e-01     3.922426271759454e-01
    2.610108728700818e-01     1.258446848187947e-01
    6.796599192370593e-01     8.716129349078633e-01
    3.099588180572767e-01     5.971375890536652e-01
    4.707300240909126e-01     2.301617543744295e-01
    3.087391356409326e-01     1.658001937016790e-01
    3.375466797581164e-01     2.370361695701082e-01
    6.387819401796546e-01     5.698496326394195e-01
    3.308533416750598e-01     6.839926438056396e-01
    7.367593500263409e-01     9.232282006803295e-01
    5.651271268836671e-01     3.678769282583592e-01
    5.405148954036093e-01     4.535166937621894e-01
    3.755475960361736e-01     8.214153363471262e-01
    2.732415352286427e-01     2.169117083137634e-01
    9.020445777062714e-01     4.629381899686872e-01
    4.244829571721223e-01     3.620862745936950e-01
    5.568716225250548e-01     5.041628146248232e-01
    8.420521993746141e-01     5.805676538316480e-01
    2.957581645122589e-01     2.292643204373134e-01
    3.715236920795933e-01     8.006819633333762e-01
    4.661676252149777e-01     6.113506887165671e-01
    5.333631304931871e-01     8.844991697375385e-01
    8.355654487737462e-01     6.339482568504627e-01
    9.139163698816739e-01     5.121667697459170e-01
    8.635445005018176e-01     3.376135831340995e-01
    5.145065446005330e-01     2.829103337475795e-01
    3.166512437467159e-01     2.080721579597591e-01
    2.322432673157254e-01     7.147904731908382e-01
    3.065850018508408e-01     7.988594150750629e-01
    4.188622959624670e-01     3.074267216050132e-01
    7.763356867082365e-01     1.278526344302457e-01
    9.334781405433524e-01     5.474548768797878e-01
    5.250883720668920e-01     4.438420756072187e-01
    4.673686126662566e-01     6.994909634466068e-01
    3.722488722006947e-01     2.105219605145060e-01
    3.245716758078707e-01     5.945669852500890e-01
    4.917182572236026e-01     3.446721196394666e-01
    2.244020873857088e-01     6.029729028679544e-01
    5.358979397308272e-01     8.950364170888659e-01
    2.945243147747885e-01     5.871884220775707e-01
    5.334059812890607e-01     4.238365712660245e-01
    3.493043050437308e-01     1.163381418194112e-01
    3.937929732339995e-01     5.500147613925264e-01
    1.483123058971159e-01     7.033744435553606e-01
    4.696410936342618e-01     3.602656751609427e-01
    2.477979815091859e-01     6.577972014267437e-01
    5.195049426012467e-01     8.186605152809713e-01
    4.651857930287467e-01     9.010976329775305e-01
    6.985268474814675e-01     5.414686803123246e-01
    6.410730541057973e-01     6.586504244893948e-01
    3.323533059321855e-01     1.167424979931817e-01
    5.068947925464515e-01     8.155149770681605e-01
    3.088983626806788e-01     6.003628929631533e-01
    3.457639431442231e-01     4.633178775082711e-01
    2.649665402314334e-01     8.509707183533322e-02
    8.689267031334160e-01     5.783262305824316e-01
    5.346197273760160e-01     3.783817340287264e-01
    4.730773931712803e-01     3.643782260144942e-01
    4.765191823158903e-01     7.429474704903674e-01
    6.214948245874844e-01     5.983175934778074e-01
    8.758573856235875e-01     5.470914640667514e-01
    3.158714927763487e-01     5.565748870464530e-01
    5.749614982243069e-01     8.737359094904500e-01
    3.741528666143178e-01     7.698491052577164e-01
    1.760987725401931e-01     8.167942563315591e-01
    4.341692709053403e-01     4.340871801377081e-01
    8.808522954289592e-01     5.792876023798885e-01
    1.853824301606777e-01     7.845835481168316e-01
    5.709874056815749e-01     8.411097178907624e-01
    6.607635472116478e-01     6.250831205766969e-01
    3.942782049526670e-01     9.035218823729938e-01
    6.795926280125194e-01     5.132919269203813e-01
    4.812505887257845e-01     3.727846788906490e-01
    3.102260919694159e-01     2.794529664451879e-01
    7.075267668139116e-01     6.382760685162532e-01
    1.696646280329336e-01     6.982248404946052e-01
    4.469062228673376e-01     6.183877244586180e-01
    4.207232404168320e-01     5.203138551966512e-01
    8.006875494664285e-01     2.598784576119426e-01
    2.423445741555791e-01     7.297270753067291e-01
    3.946687135309735e-01     5.403758944748579e-01
    8.592311452969159e-01     3.640326473117186e-01
    2.877374471151143e-01     4.703642808541937e-01
    4.592648015825088e-01     7.323333518711458e-01
    4.578417041120231e-01     6.170349455072577e-01
    6.861774349546986e-01     4.631774012275593e-01
    2.776115265543155e-01     4.935772773007915e-01
    4.811806775221190e-01     6.250280054277586e-01
    2.670791194772203e-01     5.681767813958050e-01
    6.828554325770477e-01     6.434460740816814e-01
    7.894245773038845e-01     3.609558299274179e-01
    2.887639855023016e-02     7.530260135980867e-01
    8.134345854651063e-01     6.426028156074668e-01
    6.320069305690104e-01     6.126258766052459e-01
    4.769563079089737e-01     5.531086716682484e-01
    5.468761023527181e-01     8.907470677542532e-01
    7.403545364395553e-01     9.835152637566842e-01
    3.464561412858621e-01     3.331901201231486e-01
    5.407349593327438e-01     7.498851173007249e-01
    3.705020280633649e-01     2.098085904195370e-01
    4.222672863319086e-01     2.090146847424204e-01
    7.690370859317215e-01     2.515844512023079e-01
    5.372309565976252e-01     8.002837210130175e-01
    4.340551285057372e-01     3.654959261166665e-01
    9.229212102712381e-01     5.282969738577701e-01
    7.132425041436529e-01     4.989837929541426e-01
    8.902073639054373e-01     6.047320455467837e-01
    5.166054903545635e-01     3.549041696196615e-01
    4.549033278421962e-01     9.122195204790859e-01
    3.101643852317164e-01     4.096914843159528e-01
    5.165380848514928e-01     6.579166351600338e-01
    6.840231056037447e-01     4.828102546070564e-01
    5.026432459100222e-01     8.009948218451487e-01
    4.575879234771042e-01     4.275308705415658e-01
    7.420727381255899e-01     8.629238347914425e-01
    5.107519186635092e-01     9.928273491585132e-01
    3.912178147286463e-01     6.632854018922858e-01
    1.077687773363193e-01     7.396907095806913e-01
    8.674573620282864e-01     4.190895392725175e-01
    8.498337872447288e-01     4.645855736258915e-01
    7.508176147356541e-01     6.796284062520664e-01
    3.706186455948191e-01     6.888560380837665e-01
    2.307043535575202e-01     7.456973292141368e-01
    6.731893911140775e-01     9.870402836810427e-01
    2.052228340210086e-01     8.479102179295832e-01
    7.222464948599314e-01     6.743727628323787e-01
    4.006151384045795e-01     4.881452886208029e-01
    6.582328262420880e-01     7.991375947701990e-01
    1.900302571740763e-01     7.088413324385978e-01
    3.759692732805227e-01     5.175468564966137e-01
    8.327443047664235e-01     2.063406714241978e-01
    4.621229281057134e-01     8.278333367311445e-01
    2.111142451120389e-01     6.504900344915652e-01
    6.070957750767247e-02     6.913957207638471e-01
    6.039747604634071e-02     7.059167562065938e-01
    7.624867646017571e-01     4.928246469449967e-01
    8.226602861459832e-01     6.573110269310444e-01
    4.329988048075913e-01     9.105317656107836e-01
    4.823355973461599e-01     8.819151120470957e-01
    5.198289310343986e-01     7.880337107579336e-01
    7.836522658253967e-01     4.766551654612058e-01
    2.598707934186064e-01     1.214490022696558e-01
    3.557991188963834e-01     2.677538408054689e-01
    4.189849871478151e-01     3.916977406351488e-01
    6.839185928928360e-02     7.784652868180022e-01
    5.777231844994484e-01     4.978790172668026e-01
    2.808361113374740e-01     5.807885197709488e-01
    6.353273060771685e-01     4.665588039934124e-01
    4.237008383561108e-01     7.734765940399488e-01
    1.811804566895346e-01     7.754548572393457e-01
    6.755298041750196e-01     8.706544759041484e-01
    4.006262403042176e-01     4.464867864109069e-01
    5.877028468406630e-01     4.083667494172096e-01
    7.418539456613034e-01     7.539182159671529e-01
    9.130898810755357e-01     4.742964893799515e-01
    4.644166621608625e-01     6.675188841704325e-01
    8.895341752439265e-01     2.800119828407882e-01
    5.914739979919895e-01     7.480006952922922e-01
    3.984988523538775e-01     8.158636399793687e-01
    5.428265725321532e-01     7.940519037703512e-01
    7.339398291390141e-01     7.431170956263251e-01
    7.208121579118086e-01     5.980831038794470e-01
    6.373216923495855e-01     7.961318534542708e-01
    6.853761189978574e-01     5.163112592150948e-01
    9.944989362224910e-02     6.517748647651518e-01
    7.400247211447695e-01     8.456804671513314e-01
    7.886725294864628e-01     3.243369967424590e-01
    7.972134514025457e-01     4.202529668889087e-01
    2.747125940873221e-01     6.773352561246704e-02
    3.186956116760188e-01     5.529797388847958e-01
    3.231900137987476e-01     3.087197423385564e-01
    6.479958548260505e-01     7.606589184329969e-01
    2.537086614970823e-01     4.492757169998155e-01
    8.000192241591952e-02     7.292351703045873e-01
    6.106405157975031e-01     5.080515158119173e-01
    4.890672916363633e-01     7.442558802077379e-01
    6.241523844879858e-01     8.999589840537348e-01
    6.160574566643811e-01     8.166737757998402e-01
    8.463359072924419e-01     2.737301310788141e-01
    5.487271954726652e-01     9.131484985377250e-01
    4.489888955584947e-01     9.396947617639934e-01
    4.119535668979079e-01     8.722623567780415e-01
    7.871075900550075e-01     1.665633132366479e-01
    4.138312215981472e-01     9.323969386233166e-01
    6.003024802550552e-01     9.696165210519370e-01
    4.984441019989460e-01     2.840025654356898e-01
    4.496659931554979e-01     9.495618356409056e-01
    1.766495862258410e-02     7.477386598486913e-01
    2.273918119161262e-01     5.798471497674055e-01
    7.343010556355630e-01     7.757161792843292e-01
    3.348889906027106e-01     1.095720715654210e-01
    8.080744384670722e-01     1.686589899478447e-01
    2.072109458404627e-01     5.978325624755811e-01
    5.058170103674500e-01     8.139184727236013e-01
    8.302353090035651e-01     5.479177265597327e-01
    7.750816000340066e-01     6.806552157190157e-01
    3.066309242748350e-01     6.384601005096734e-01
    7.491683606595434e-01     9.245732975306987e-01
    4.426535625744972e-01     7.827978217021977e-01
    2.068049515041936e-01     5.750622407105345e-01
    2.806396665778470e-01     3.452162812111942e-01
    7.277450654673726e-01     7.903901168898153e-01
    5.217765644558592e-02     7.557637122607579e-01
    4.589009259326068e-01     3.269140836846824e-01
    1.727645566264079e-01     7.734289032266821e-01
    8.374944571011392e-01     2.553202400874228e-01
    4.756960969443833e-01     8.608453550541634e-01
    2.529603084822567e-01     1.358214965597877e-01
    3.509419586707425e-01     5.694714529446983e-01
    4.159893542516172e-01     4.805871225243319e-01
    1.327911228729992e-01     7.640985039997347e-01
    2.103702634026462e-01     5.501492313553359e-01
    3.174856776171471e-01     7.466591903977909e-01
    6.698163295767942e-01     7.425757496186075e-01
    7.334316163707132e-01     8.198220679216621e-01
    5.203115959650366e-01     8.571746765870825e-01
    6.414928857614984e-01     7.360466834519542e-01
    4.079380826828001e-01     5.845310015269245e-01
    9.354309635705599e-01     5.458459467951523e-01
    4.051488521031416e-01     5.595701385828731e-01
    7.987799291294112e-01     6.913911485283608e-01
    9.065509403551557e-01     3.641576175970016e-01
    4.646370844990135e-01     7.750983475624187e-01
    3.753397087036418e-01     2.426428728051251e-01
    6.379785913500222e-01     5.848881030658243e-01
    5.640562420044585e-01     3.633592541897489e-01
    7.036260187283156e-01     6.373584874782522e-01
    3.236799089846988e-01     8.294570803563925e-01
    7.040648361672718e-01     7.495246932856192e-01
    7.386001741464107e-01     7.918462775550587e-01
    6.439752286169212e-01     9.949649568928869e-01
    3.943864185779725e-01     7.808643483639550e-01
    6.473929708008376e-01     7.263860773746424e-01
    2.989615549783169e-01     4.147987811162789e-01
    6.053366863767159e-01     9.845355043982306e-01
    5.174870916146361e-01     6.613219731109894e-01
    2.775997448693651e-01     6.241414642572507e-02
    2.664633813860460e-01     3.875384371218042e-01
    6.917086931424565e-01     4.954261165722997e-01
    4.358821583896502e-01     3.540661631987171e-01
    3.060227848592708e-01     1.049024933763689e-01
    4.461201721261115e-01     4.574414926919647e-01
    4.983942508778665e-01     4.861196810476395e-01
    5.719447754145190e-02     7.742064255171274e-01
    4.393831148200184e-01     7.904028900081996e-01
    2.694244835398468e-01     4.954139295614264e-01
    4.652029694936965e-01     2.588781938259040e-01
    2.826291051497231e-01     3.823793190610070e-01
    5.167694251399574e-01     4.945994436567475e-01
    3.492982001111203e-01     7.358030377514977e-01
    6.673045694922554e-01     4.265739648383806e-01
    3.399118409620544e-01     3.478713060088267e-01
    2.694842963996363e-01     3.394075751305420e-01
    4.753309685510484e-01     2.846119147970071e-01
    5.314761115347911e-01     3.250556315506408e-01
    5.582485891370134e-01     4.910334320663287e-01
    5.827345616676172e-01     7.367211845591315e-01
    4.739819003375065e-01     3.104215667938557e-01
    3.235173786532620e-01     1.089466629576463e-01
    8.043466505340537e-01     3.141280532793752e-01
    3.003410631570418e-01     3.088288881540615e-01
    2.845970219000307e-01     4.701163880237565e-01
    3.567740614691390e-01     1.490442921249507e-01
    5.555961136486021e-01     4.057820780508268e-01
    5.174000306005013e-01     9.265273999976401e-01
    4.048493248575833e-01     4.945135638947995e-01
    3.895309265489599e-01     1.756631326327635e-01
    7.779193189367717e-01     3.645304025231186e-01
    3.252625946825871e-01     8.548142659850273e-01
    3.618779801331311e-01     7.699632354199234e-01
    4.169889048385435e-01     2.753350855091692e-01
    8.026935398466633e-01     2.076702492179975e-01
    4.714053164936335e-01     3.760860418377476e-01
    7.719985772341525e-01     1.934828869543743e-01
    3.372173566734931e-01     5.892351602945951e-01
    4.443805617617799e-01     6.519400013674165e-01
    6.996223987292943e-01     9.424284030976806e-01
    7.695269757293465e-01     5.862202540063932e-02
    3.434032531704594e-01     7.496449127469813e-01
    7.585682517701616e-01     3.455515149348376e-01
    9.523532166393052e-01     4.413310434312772e-01
    8.737010936632372e-01     5.436140599017819e-01
    7.861674239166178e-01     6.981954474366828e-01
    7.038927794845706e-01     7.876262257129018e-01
    6.302900940402524e-01     7.018237545176793e-01
    7.253294728414366e-01     6.548313122988200e-01
    1.811068135919811e-01     7.621550427800234e-01
    2.301702201764285e-01     6.397910532774671e-01
    5.430863677282305e-01     5.026546115814097e-01
    2.736044330859309e-01     4.009019477847907e-01
    3.448862029828218e-01     8.841187994812353e-01
    2.717844446091836e-01     3.249240680830646e-01
    2.946272092946661e-01     1.986957200121402e-01
    3.958203995622092e-01     4.045686940091856e-01
    8.732328173594383e-01     5.745818074448061e-01
    4.148454527590326e-01     3.373616248437433e-01
    8.622337027634766e-01     5.111529838950716e-01
    7.109072646890631e-01     6.989283497055724e-01
    2.905454352187095e-01     7.348493140114352e-01
    8.506693153974167e-01     2.920200147740202e-01
    5.451784124246887e-01     8.036578758030468e-01
    5.824774689944020e-01     7.881537602554132e-01
    6.129572183103226e-01     7.521588706911879e-01
    5.463912325304029e-01     8.470447954224599e-01
    7.565919743643086e-01     7.147554980176303e-01
    7.491253587515919e-01     5.163471257535833e-01
    4.853515853091559e-01     4.169606806431426e-01
    3.572231028604650e-01     7.830551768233971e-01
    3.066716705029646e-01     6.111013866739585e-01
    7.227571662731268e-01     9.911317676958610e-01
    2.740204131093534e-01     4.548921426519239e-01
    8.697497135860486e-01     5.534971580237918e-01
    1.914057448880923e-01     8.123734540163453e-01
    5.809772368695357e-01     6.622754909550803e-01
    5.123224555288036e-01     5.357588473761778e-01
    3.283806343609720e-01     1.109636451153941e-01
    5.599524842643051e-01     5.065252194378371e-01
    3.152770921869678e-01     3.831866252623032e-01
    9.776787580227031e-02     7.114653513132948e-01
    8.101988275940121e-01     2.659176825957311e-01
    2.916333273932152e-01     7.296833581007515e-01
    4.837867374283663e-01     9.068940192600485e-01
    8.096674029596221e-01     4.956442000090313e-01
    3.480439725752685e-01     2.789579261611038e-01
    3.175550626170568e-01     8.692553115285430e-01
    2.679801341141905e-01     1.610333385195887e-01
    5.293608483675304e-01     8.880351764360825e-01
    5.709297800342172e-01     3.336605159455279e-01
    5.957706863002340e-01     7.644544817792044e-01
    2.801486981417567e-01     8.850298696008859e-01
    3.440627620422348e-01     6.608693917429296e-01
    9.345937695271451e-01     5.644569855720498e-01
    1.526012340071043e-01     7.346450774975058e-01
    1.433884773091505e-01     7.868972835094539e-01
    1.354850939794179e-01     7.672004945531493e-01
    3.396442788021760e-01     2.535652818144621e-01
    2.656718304734046e-01     4.701352304103892e-01
    2.880767871113563e-01     7.191806889856578e-02
    6.496445707157613e-01     4.182036711807919e-01
    5.853797989691775e-01     7.499363793493969e-01
    5.039780333480017e-01     9.981330185062539e-01
    5.045987445349062e-01     6.038138539645168e-01
    4.466851229375814e-01     5.353092083274029e-01
    4.501572321062197e-01     7.322411024049915e-01
    3.304733507890923e-01     6.239016308044696e-01
    7.577467131084138e-01     7.234606549356295e-01
    4.531222876645932e-01     5.906239413459158e-01
    2.595226292003835e-01     3.037922048274343e-01
    8.277066888993638e-01     5.227284943629165e-01
    8.237995419895549e-01     5.837267650049517e-01
    8.465363741336825e-01     4.507122173392887e-01
    5.588264555934643e-01     6.816460602958697e-01
    2.259074465989045e-01     6.732569433408740e-01
    4.700002370371912e-01     4.255659695400479e-01
    3.261701700824348e-01     6.030505784034643e-01
    8.869251062472050e-01     4.476155894248361e-01
    5.684625286667994e-01     9.922948374395424e-01
    7.118205094421546e-01     5.616921453557102e-01
    3.056683861748610e-01     4.229827990480199e-01
    6.526648359570326e-01     4.578686723211778e-01
    4.279415832706522e-01     5.249461074483750e-01
    2.159895810385377e-01     8.253669348146777e-01
    8.069660940099144e-01     1.537845113829678e-01
    9.174139183866792e-02     7.301165246342964e-01
    6.363967417154718e-01     6.528142923663341e-01
    3.863431462636330e-01     7.783556623907459e-01
    5.253585340978246e-01     3.793068466656658e-01
    8.941951803901032e-01     3.133260246406698e-01
    1.991562898509678e-01     5.607748610306024e-01
    7.088626244580976e-02     7.410149987172786e-01
    3.599564825186744e-01     9.100962185105810e-01
    7.692751078745532e-01     5.697260393272010e-01
    7.307205502935605e-01     9.144388049486074e-01
    2.895234017577329e-01     7.163903629497815e-01
    2.610215034707407e-01     1.832037647721789e-01
    4.238770370992564e-01     2.942580753266195e-01
    4.148817860517053e-01     3.080921110310402e-01
    6.214553758576645e-01     4.625519263860314e-01
    3.480446552414019e-01     4.509371860237260e-01
    4.770275366549873e-01     3.489070787778241e-01
    8.692115906283897e-01     4.212399793534860e-01
    3.083802032744550e-01     6.954868252317037e-01
    5.689070236935886e-01     5.399417247750770e-01
    3.865412251480456e-01     6.963306071923461e-01
    6.124358143644487e-01     3.939838906072259e-01
    9.040174367513402e-01     5.464054817009029e-01
    8.273182074406960e-01     6.405471688720074e-01
    5.726107095912155e-01     9.657732999119530e-01
    2.420430527342325e-01     6.818563575473444e-01
    8.259419867464919e-01     5.915990093233803e-01
    3.867908841533195e-01     6.979176492259863e-01
    5.600057873677602e-01     9.193244710836062e-01
    5.089395684454801e-01     4.279648462230381e-01
    3.199087990198418e-01     2.684603697029491e-01
    4.082269842078824e-01     3.023662285177374e-01
    4.197588912379838e-01     7.319911024490687e-01
    4.803587777385856e-01     4.020591804814105e-01
    2.409509920009524e-01     6.816031574232518e-01
    1.238156095416522e-01     7.437239887715530e-01
    4.933363763757863e-01     7.185140563963553e-01
    4.350891472880886e-01     4.582876528185348e-01
    8.730635826120406e-01     2.491285885352756e-01
    5.468939466867445e-01     9.236980375637072e-01
    3.063303060696515e-01     6.073808133168686e-01
    8.262657044305464e-01     5.973731332314285e-01
    6.298171919336591e-01     7.201414898762953e-01
    6.988071873274060e-01     5.381138271985048e-01
    5.829317238646863e-01     7.456381837703353e-01
    4.518710401634835e-01     5.756066881324182e-01
    3.635063626997889e-01     5.296544279484218e-01
    7.101938400035378e-01     4.960199749644983e-01
    6.957579572499301e-01     6.524916310260949e-01
    7.726010572379506e-01     7.259946744598659e-01
    6.923043898567273e-01     7.130508270801463e-01
    7.390565823041323e-01     8.260234902745122e-01
    3.305748060360811e-01     1.039623280448290e-01
    2.906555510610304e-01     3.961813546762575e-01
    8.204351781336384e-01     2.206646664305798e-01
    3.015045557346200e-01     1.036276416645989e-01
    8.232991608970078e-01     5.663398299037353e-01
    4.095767724265968e-01     1.747346008528533e-01
    6.884523446601433e-01     5.809619576719123e-01
    7.952093346521339e-01     1.624379070186812e-01
    6.645683072416593e-01     6.430815950039219e-01
    4.141795559224679e-01     3.126907089865750e-01
    3.282384136114249e-01     2.341416051871086e-01
    2.809288421453831e-01     2.448461185622359e-01
    2.595666748130659e-01     7.046120329630904e-01
    2.587474405771137e-01     9.725350562193102e-02
    8.692097417984329e-01     6.208852962081743e-01
    2.555212523461092e-01     5.195583890866868e-01
    3.467687815653742e-01     5.275402265621588e-01
    6.312501550854597e-01     9.335078180916440e-01
    7.653789052867560e-01     1.975967506194332e-01
    2.762063548784256e-01     1.030190159048222e-01
    5.055575656571224e-01     8.453336374214472e-01
    5.299411976181327e-01     7.370458863701063e-01
    1.980161776195941e-01     6.562406888645905e-01
    7.940769634105878e-01     2.100095818299456e-01
    6.352674926154318e-01     4.933153978922029e-01
    7.786051458364234e-01     6.331635409057962e-01
    4.354888099796549e-01     1.989656070891757e-01
    2.919793963035975e-01     2.083984773906095e-01
    7.480142690113354e-01     6.867537744883930e-01
    9.001657741937888e-01     3.565534509379358e-01
    2.676826026472600e-01     4.988718905343488e-01
    5.992331925132783e-01     9.695326548806038e-01
    2.861189798848008e-01     3.046481401294032e-01
    8.213757586056605e-01     4.739386554802277e-01
    7.564347116333272e-01     5.609573428105158e-01
    6.302530784440136e-01     8.267027047412103e-01
    3.234875748617867e-01     8.038836751673044e-01
    5.469494765766246e-01     6.267130768431077e-01
    5.678510105848326e-01     9.588422838352414e-01
    6.074051903156252e-01     7.011357431505637e-01
    2.546664092825085e-01     1.400409550453973e-01
    6.498866361679274e-01     8.025999713906853e-01
    6.024252197121849e-01     9.245608551029758e-01
    5.311189647713629e-01     6.325859572623409e-01
    5.497275319524030e-01     5.307623110339477e-01
    2.743543970166497e-01     1.626320822088189e-01
    4.706775258755318e-01     5.474817775666275e-01
    2.736354982457895e-01     1.446545997070241e-01
    8.072578171308310e-01     3.088495392302104e-01
    2.753891195840765e-01     4.991029712501132e-01
    8.097065749096016e-01     3.299405848381766e-01
    7.195556691110427e-01     8.979161723159147e-01
    8.290303576540273e-01     1.981143441920169e-01
    6.731443366796021e-01     6.555275142649218e-01
    5.159811965618311e-01     8.166092382183968e-01
    4.192859629796256e-01     5.942364949626896e-01
    3.796986858413934e-01     1.379236462272037e-01
    6.733578582723384e-01     6.175386875085213e-01
    1.287833523044889e-01     6.910660006014731e-01
    3.628238235913882e-01     6.090172269441214e-01
    7.105180130269066e-01     4.617844527725770e-01
    6.154359866495559e-01     9.633468557284260e-01
    4.581728269960617e-01     6.727566309024823e-01
    8.030759833632244e-01     2.855916249829371e-01
    4.236832746658596e-01     3.661052743792723e-01
    4.127067776606774e-01     2.381262841985300e-01
    5.833547810645417e-01     8.132774680488715e-01
    3.702182675941535e-01     9.210342528726573e-01
    3.731508914686491e-01     8.245126800167361e-01
    7.804593046096568e-01     6.899367005179832e-01
    3.173214489636771e-01     6.197513270171021e-01
    4.091315719523497e-01     4.979165731071263e-01
    2.944875979324832e-01     5.406007939107095e-01
    3.532809813057247e-01     1.394341895524074e-01
    6.866866678215114e-01     6.513867254164019e-01
    6.595055460090108e-01     4.327875740681321e-01
    1.608575645069372e-01     6.618662373781933e-01
    1.873743980720235e-01     6.832602907902899e-01
    5.644646178841877e-01     7.482388146421853e-01
    4.471418170859046e-01     7.815420825975382e-01
    7.828495208870399e-01     1.683135238571424e-01
    2.564657555315970e-01     5.571809667378965e-01
    2.793732477132693e-01     4.698821605912943e-01
    6.199733973163201e-01     9.476923541754173e-01
    2.818622564515315e-01     4.569042555095179e-01
    5.021460416692911e-01     7.222374177855780e-01
    7.549138095717968e-01     1.764129558824277e-01
    8.557510182004737e-01     3.024704596863743e-01
    2.182306573594066e-01     6.916364112358704e-01
    4.049735218513032e-01     8.202667742454073e-01
    9.062760346680518e-01     5.597011085315342e-01
    5.471710137383062e-01     4.043830175684631e-01
    5.163314965896003e-01     5.662850328730743e-01
    2.974907572928720e-01     3.860857080160084e-01
    1.602566234428444e-01     6.158510516282892e-01
    8.218130276341664e-01     3.154805906414628e-01
    5.267347758310837e-01     4.909806990126950e-01
    9.750373989948512e-01     4.669782078336930e-01
    8.269892851243312e-01     6.342079329245043e-01
    4.067825291156326e-01     6.632839216960511e-01
    7.160607465593685e-01     6.638552671531033e-01
    2.359693117961122e-01     6.539134310392131e-01
    3.220321573862900e-01     4.486606980411602e-01
    2.233299781868786e-01     7.802946071693072e-01
    8.743955117618852e-01     5.807817073569232e-01
    7.102402491363637e-01     9.185482506754770e-01
    4.144764507196851e-01     7.775896511873794e-01
    2.771353321629005e-01     8.426112386128903e-02
    4.998274257635756e-01     5.505160995522587e-01
    3.415197542124606e-01     6.209064123464746e-01
    2.577823226072569e-01     4.437287956183228e-01
    2.612433152541437e-01     6.313327171386038e-01
    3.010808149985389e-01     3.633861287329717e-01
    6.344039429052507e-01     8.833330682496737e-01
    4.095402621827968e-01     3.027025661387586e-01
    8.387045054155575e-01     4.966968248186727e-01
    7.923828612205536e-01     3.118145158984938e-01
    5.809563764929585e-01     9.832043843895421e-01
    4.877318236220377e-01     7.515462483581655e-01
    8.288105978538254e-01     2.181569544799500e-01
    2.726928317729109e-01     5.792438105975838e-01
    4.598810345697313e-01     8.250716716963905e-01
    4.671638510895160e-01     5.119105627526320e-01
    7.412019669760297e-01     8.838963099029319e-01
    7.685060021835274e-01     5.722453196585021e-01
    5.593745888503994e-01     4.701749860128274e-01
    6.703576058221382e-01     5.335504279405606e-01
    8.282126192126894e-01     4.538045003108552e-01
    4.112210547972808e-01     8.998286718387901e-01
    1.886381637255956e-01     6.571619679744781e-01
    2.453393811000592e-01     6.312129175262743e-01
    6.267892335047568e-01     5.714466101879496e-01
    3.878057565718707e-01     1.468449837500611e-01
    8.111297573881801e-01     1.548657806528534e-01
    6.721077395856849e-01     5.815730107879776e-01
    8.542108621294715e-01     4.857759357518606e-01
    4.687143456671862e-01     5.006944053516234e-01
    2.735232796445854e-01     4.943762743623040e-01
    4.414633521648610e-01     7.619767056711256e-01
    3.908170326970227e-01     4.587114972533537e-01
    1.223245245758839e-01     8.046805854505945e-01
    5.204929936318203e-01     3.752388797070015e-01
    7.410563431958773e-01     6.573830985916607e-01
    3.343072682419603e-01     5.238974330114934e-01
    5.772368198134772e-01     9.952286478713289e-01
    2.675635110715567e-01     4.007824793371173e-02
    1.169016333594679e-01     7.627733048674875e-01
    6.414286035903368e-01     7.517840750550357e-01
    5.113099146738053e-01     5.183586886093371e-01
    6.290250272367596e-01     8.986145698362655e-01
    6.450617661751951e-01     6.619143012082874e-01
    5.327760653185161e-01     2.939751030789887e-01
    6.795604118803613e-01     9.533560897567198e-01
    3.838439742192381e-01     5.169336021884084e-01
    2.891789877633928e-01     4.041578935612589e-01
    3.647269523849223e-01     5.730072147765793e-01
    8.527326890561344e-01     2.407213514429184e-01
    1.457311159751615e-01     7.132159378429310e-01
    7.269496796270861e-01     5.518301248777057e-01
    7.702288537281980e-01     6.663690913037167e-01
    7.634845707678269e-01     4.800008389617628e-01
    2.559491919000040e-01     7.214020646178607e-01
    4.985517035257554e-01     2.716480209548470e-01
    3.226741986765213e-01     3.492558103473991e-01
    5.183072094741941e-01     9.880984048493913e-01
    4.133364057630543e-01     8.901075215254488e-01
    7.435844333849385e-01     5.225851454773655e-01
    2.063165144871859e-02     7.385153652421261e-01
    7.091387958233336e-01     8.398948932194631e-01
    5.100909022613316e-01     3.182722652052015e-01
    2.807452976921190e-01     3.374324180186286e-01
    7.805877933438951e-01     3.804259185302340e-01
    4.066130586058809e-01     7.502242778891378e-01
    2.778017632514741e-01     1.431985017746567e-01
    6.227974543012984e-01     9.700998587448216e-01
    3.858053867859311e-01     5.496874763982550e-01
    3.115421357587373e-01     8.709016906100815e-01
    5.851242036164136e-01     3.509873687418855e-01
    9.426544193171549e-02     7.537612110006081e-01
    7.975338267323598e-01     1.988752419766685e-01
    5.389394811737680e-01     4.019244389485949e-01
    5.456648351601484e-01     4.061294463558075e-01
    7.333684581494792e-01     8.764905405896261e-01
    5.617743249178535e-01     8.387549810830118e-01
    5.580757306155052e-01     3.553573490462564e-01
    2.726392886428729e-01     1.440099109038568e-01
    6.526274147999533e-01     5.281442146599760e-01
    3.227571165244428e-01     5.234221149551316e-01
    7.870051876619160e-01     4.704355917194920e-01
    6.088635182156342e-01     3.817649655620510e-01
    4.412640621822527e-01     8.706345823411933e-01
    5.810158885523342e-01     5.776859060024253e-01
    3.373105270836806e-01     8.880780143125236e-01
    3.522188319979801e-01     2.067962089918723e-01
    4.652225932524459e-01     8.098706757418815e-01
    8.282668205890648e-01     4.477878059632393e-01
    1.595337213790140e-01     7.324198113431499e-01
    7.857719009729400e-01     6.418650524547053e-01
    7.923551562005106e-01     1.944185332356601e-01
    7.662790045061125e-01     4.342710220591265e-01
    6.778830566244849e-01     7.019544385049722e-01
    2.598799146252689e-01     2.860166190670431e-01
    6.346350565222829e-01     4.902937026802979e-01
    2.600243439763499e-01     1.500215518811953e-01
    6.477080321347555e-01     7.623475912032326e-01
    4.377997412416852e-01     2.597784039707290e-01
    7.692790640206321e-01     6.880679494137615e-01
    2.218663584734732e-01     6.118173028935984e-01
    5.598205056818721e-01     7.390797823709239e-01
    9.097992419841159e-01     4.790975269037384e-01
    3.546534644462833e-01     3.447033481535423e-01
    6.659925908883143e-01     5.854291448900082e-01
    7.930160811917556e-01     2.175823353347685e-01
    8.450494337839470e-01     2.642433076315791e-01
    8.033673757167957e-01     2.166620280646242e-01
    2.525312127690835e-01     3.062009589276858e-01
    7.466393337033402e-01     5.653644699699156e-01
    5.027211451470357e-01     3.973033138451593e-01
    4.921490550271788e-01     9.673349175656371e-01
    5.653427814965953e-01     5.567452533830193e-01
    2.247261825163279e-01     6.017849091810785e-01
    5.570636517913438e-01     4.597530425652823e-01
    8.393625972836259e-01     2.252707154377206e-01
    7.263032471525227e-01     5.632908567511019e-01
    7.195505754498930e-01     8.722045291486098e-01
    3.231299991634686e-01     3.235969313964330e-01
    4.474393361566303e-01     6.759741166572426e-01
    8.218873337460265e-01     5.125182018208512e-01
    4.036036089384840e-01     8.240939751933238e-01
    8.652589251419307e-01     6.319121302535179e-01
    2.377634243286970e-01     7.349244926219373e-01
    4.110035628971650e-01     2.003798231179656e-01
    5.873727917078237e-01     3.451059099868854e-01
    6.884445245812129e-01     7.597359704832687e-01
    8.089737147058635e-01     3.329050389310396e-01
    3.859117083285797e-01     4.719981096810592e-01
    7.705415538378747e-01     6.203529845346663e-01
    8.843393631884524e-01     5.016032248223862e-01
    1.848675571208254e-01     6.709354878688496e-01
    4.572885122148949e-01     8.879486140425171e-01
    3.031604978976765e-01     6.141718744762797e-01
    1.618355971098180e-01     6.943628087182022e-01
    2.694729465407461e-01     6.025678666849610e-01
    9.356732747462656e-01     5.373389177350825e-01
    1.731698028284523e-01     5.782718526051528e-01
    4.491874464896279e-01     6.221334449719237e-01
    7.175578432588369e-01     7.313002754961415e-01
    4.701587393265109e-01     9.542350479162823e-01
    6.844763105541305e-01     6.909188676742533e-01
    6.600670530930702e-01     7.329854459222162e-01
    2.361923603009600e-01     8.659386603858612e-01
    8.405657201629291e-01     3.229124444474675e-01
    2.642489338466673e-01     1.945247066848033e-02
    1.723607016522833e-01     6.216661103201472e-01
    3.163454747496383e-01     4.829568282179156e-01
    9.174824225737545e-01     4.177693508242402e-01
    4.145068768922847e-01     3.025758421414794e-01
    3.086185832181978e-01     8.723136701803154e-01
    2.402802229781127e-01     8.700082086028480e-01
    4.692385170364032e-01     8.754313538645174e-01
    7.302869306834558e-02     7.726557824387887e-01
    2.753847356469437e-01     2.385437845634929e-01
    7.501526661653881e-01     6.752324122851306e-01
    1.030082887364607e-01     6.598497745818374e-01
    2.695837303506332e-01     3.162882515095509e-01
    5.754408733988627e-01     8.320994790514311e-01
    5.235724689933858e-01     8.753374350930241e-01
    8.370844665942312e-02     7.573922674955097e-01
    3.816408800543130e-01     2.697557712022981e-01
    7.433242182881844e-01     7.243119957053444e-01
    2.467000760750871e-01     5.843555510034811e-01
    4.537358304288708e-01     7.628579041884821e-01
    3.311251881289764e-01     5.707611147700759e-01
    5.788240858730015e-01     9.747517393069041e-01
    4.102224258549979e-01     2.701434761025996e-01
    8.213154529179360e-01     3.490085937463108e-01
    6.649478181899314e-01     8.024725757503595e-01
    2.413893623208793e-01     8.550711037609405e-01
    2.657473725752708e-01     3.913870026934659e-01
    2.773913816678519e-01     8.388981561507725e-01
    2.752598406906653e-01     1.182426309887240e-01
    9.103412568040001e-01     5.628625823978106e-01
    2.735147639634771e-01     6.756233332867390e-01
    4.875094611569191e-01     8.889648205282362e-01
    4.380731226996737e-01     4.721544882496875e-01
    4.631729897394158e-01     7.996446358595222e-01
    3.439110632188414e-01     9.072613937969594e-01
    2.783951955258127e-01     1.297308063485524e-01
    5.903431507380930e-01     7.303429749392020e-01
    3.551955714888798e-01     8.297082588831378e-01
    5.305202752117746e-01     9.130751060626808e-01
    7.039509667906678e-01     9.145264871101675e-01
    5.519913585682660e-01     8.287897254806852e-01
    9.415343611561710e-01     4.304437485889424e-01
    2.787501442875101e-01     5.250838746824965e-01
    3.375156506275697e-01     1.717245603439923e-01
    5.842094313414192e-01     5.977406080551554e-01
    2.900513341450787e-01     2.654367006572211e-01
    5.305719837995444e-01     7.888879062297069e-01
    6.158000921837952e-01     6.164738873442587e-01
    5.850588448182283e-01     5.772605316171356e-01
    6.754580164928794e-01     7.641303802217828e-01
    8.079449863044504e-01     4.170554816793085e-01
    3.797502948295624e-01     1.588957252526437e-01
    2.006246102252847e-01     6.326580828651481e-01
    4.176888824534216e-01     8.411720321669861e-01
    9.025875310254363e-01     4.905295780343305e-01
    7.001849072226475e-01     8.370392651030353e-01
    2.819802789617963e-01     6.693519758153385e-01
    8.725155226723581e-01     3.495559701854782e-01
    8.025051861058952e-01     5.752840460935228e-01
    4.092886210449974e-01     6.260410594958057e-01
    1.032378325470812e-01     7.029839871195412e-01
    4.526176012883820e-01     9.148433579581560e-01
    3.521741499589033e-01     3.096136160668583e-01
    6.562890808575981e-01     8.084635482265627e-01
    5.188906079926241e-01     3.656860414073467e-01
    6.459945927519926e-01     9.301501120451009e-01
    6.906116006292095e-01     4.931646884047675e-01
    3.150889129992726e-01     7.740008197187940e-01
    2.835395524413848e-01     3.387897556032530e-01
    5.062823397287902e-01     7.091126139457953e-01
    2.964582519611156e-01     6.758500537015182e-01
    2.278022669683564e-01     8.604960931340508e-01
    2.975218189128215e-01     8.858867420858999e-01
    6.944422984951558e-01     7.490944692439989e-01
    3.245165254010201e-01     2.909227151914355e-01
    5.217621148963772e-01     8.743752136249228e-01
    3.408740994443694e-01     4.125339352688506e-01
    6.432106946752018e-01     7.663434314335339e-01
    2.722722315713030e-01     9.875283509247645e-02
    8.419988546950835e-01     2.589145436956606e-01
    7.630997864084846e-01     4.694335753553585e-01
    6.871784862534110e-01     9.522950071689350e-01
    1.888836596149472e-01     5.919826622635331e-01
    4.346954007860637e-01     3.512341899370292e-01
    4.711873696387875e-01     3.897314062973136e-01
    2.944369949233551e-01     3.925851948704862e-01
    2.655172089679620e-01     3.910003029456597e-01
    6.909489473521893e-01     4.715723226972446e-01
    3.044067691075224e-01     4.758933531060433e-01
    4.370495411892255e-01     7.321883565019369e-01
    4.535630133424802e-01     8.680775559783659e-01
    3.609519955367775e-01     9.036069820544304e-01
    3.533342427841161e-01     7.422652782807375e-01
    8.702825151534479e-01     6.181992244775053e-01
    8.345889196109070e-01     1.971010796168878e-01
    4.495410817886917e-01     3.190248134109717e-01
    2.570527552443934e-01     6.294092203066746e-01
    7.831117046158828e-01     2.816721502483411e-01
    8.318334498352070e-01     4.513527492966464e-01
    4.656882791533734e-01     5.951381527897306e-01
    4.205563775837804e-01     5.164259871240399e-01
    6.422345939245102e-01     9.587445272883138e-01
    5.738897475818087e-01     8.934467155530272e-01
    3.335107580297643e-01     2.740462034624555e-01
    2.640912789253211e-01     7.766339252628705e-01
    4.079329306795100e-01     5.585517115999963e-01
    5.806008512426444e-01     9.826115605304032e-01
    3.175562917660220e-01     2.159899427167141e-01
    5.744599897444111e-01     3.745758182886880e-01
    3.488023545131042e-01     6.204223743619171e-01
    5.857747432561906e-01     7.974173106577436e-01
    7.982846834154341e-01     1.462678753380002e-01
    5.836941525618315e-01     5.940852672935009e-01
    1.394617304811835e-01     8.099959898129200e-01
    5.149872317125945e-01     8.298214486525799e-01
    2.856842598475556e-01     2.344811513417207e-01
    4.678880034351435e-01     6.947522711396261e-01
    2.694284673073259e-01     9.139432270547698e-02
    3.803154530403449e-01     4.737557985723433e-01
    4.310016357657638e-01     7.229922332236727e-01
    3.472969427045541e-01     7.893723231050436e-01
    4.482687102463250e-01     2.619583736111749e-01
    1.016923095512212e-01     7.491477565736889e-01
    2.796694966706150e-01     1.666003815078253e-01
    4.602035290740756e-01     7.357179847220532e-01
    3.896923445840179e-01     1.760165989019778e-01
    7.703055337084339e-01     3.770059277611776e-01
    4.145155980231493e-01     1.994532411658370e-01
    4.771726592649980e-01     3.416318967737020e-01
    6.219852038744957e-01     6.047785781297179e-01
    7.165443140638346e-01     7.876591585937081e-01
    3.286706303814625e-01     5.670977404829989e-01
    7.950247917492500e-01     1.301776303850464e-01
    3.116188342214850e-01     3.768475058403784e-01
    4.770801269226101e-01     3.729919155930810e-01
    7.022919104974649e-01     8.311681988184496e-01
    3.685189040895797e-01     5.291852015667502e-01
    4.902722880914019e-01     4.962437321347150e-01
    3.822778121017565e-01     6.417301097401927e-01
    5.305102187419511e-01     3.449610952287639e-01
    7.727825845094835e-01     3.914706836636024e-01
    4.166183944277725e-01     8.088527656997568e-01
    2.646947172893405e-01     3.419103220622383e-01
    1.900507906235743e-01     7.546684908261596e-01
    7.082999475979883e-01     5.662626700463879e-01
    4.086741102194567e-01     5.381540434083923e-01
    2.576788828317716e-01     6.788020455153166e-01
    7.526394304992112e-01     1.488563899604424e-01
    8.264584823832249e-01     3.674524029582141e-01
    9.450205914415699e-02     7.724332110771466e-01
    7.290728378679577e-01     8.685468526126399e-01
    6.980376622528353e-01     7.124797130894136e-01
    2.772421497351280e-01     3.218937826707990e-01
    8.264894887672306e-01     3.696953763485876e-01
    4.538134081342549e-01     9.228110949224302e-01
    4.074525954258060e-01     6.144994729417594e-01
    4.294060919497673e-01     5.889605588568857e-01
    2.258632998670371e-01     8.369987744152962e-01
    6.541828316592099e-01     6.553554221385601e-01
    3.547383268330304e-01     9.252604670852984e-01
    3.523614735110864e-01     8.529421650564476e-01
    6.278145468117573e-01     5.620710374124132e-01
    4.174458726529059e-01     1.823964621902336e-01
    6.433734996454979e-01     9.826211100718542e-01
    4.999861721375769e-01     5.515886405464047e-01
    3.983526830014161e-01     2.475976497188573e-01
    4.018238149887036e-01     8.969461339871121e-01
    4.818769120105270e-01     7.946017822094881e-01
    2.592258392823164e-01     6.238421541595984e-01
    5.093927592875060e-01     9.122442943342767e-01
    4.295854799551214e-01     9.558193766370608e-01
    3.577441285619132e-01     6.992998217151276e-01
    2.714167940573413e-01     4.190327896279713e-01
    5.764643235996073e-01     5.179344056548882e-01
    4.555580415238196e-01     8.049509495647775e-01
    6.529327918031907e-01     8.310690096315385e-01
    3.428255696127122e-01     3.533998249036754e-01
    2.619297307477673e-01     7.782041994564863e-01
    6.422893039633826e-01     9.184501321663950e-01
    2.598196041022360e-01     5.800041006288170e-02
    4.837851869762431e-01     3.734353878208000e-01
    5.766487770888633e-01     7.794625072211949e-01
    2.451605949850346e-01     7.161220228815689e-01
    7.756543744955100e-01     3.452697549253667e-01
    5.461363066430639e-01     3.315330937064969e-01
    6.838851411120788e-01     6.649673689598550e-01
    9.352166928132287e-01     4.559437308582917e-01
    5.024900903411944e-02     7.346758445280579e-01
    8.704887217303681e-01     4.929152191082112e-01
    4.376106485967677e-01     4.675369781599367e-01
    7.493836310827049e-01     9.330927814526643e-01
    5.690513828435837e-01     4.866706787200996e-01
    2.570841146492993e-01     6.637353728100156e-01
    1.390139761142218e-01     7.870459918441775e-01
    5.232680876412719e-01     9.080801076597992e-01
    2.072964175472772e-01     6.067825100079556e-01
    6.127333764352068e-01     4.386394313667723e-01
    5.258677043110702e-01     6.413232232741450e-01
    3.387794481964445e-01     1.922289408614598e-01
    3.813924399360947e-01     5.190857366221024e-01
    2.383006343266968e-01     5.494651177457006e-01
    4.454272628989651e-01     5.848141863991589e-01
    4.461821056841605e-01     5.539018895283930e-01
    3.933019161758293e-01     7.208704527812996e-01
    4.539157844087478e-01     5.388527408597750e-01
    6.890736903961440e-01     9.802410726486241e-01
    2.336032815522523e-01     6.533434507878734e-01
    2.392003686194123e-01     6.845095450411499e-01
    6.993925008297401e-01     9.288270260910131e-01
    7.780039155490359e-01     1.165736844190470e-01
    7.938578724608208e-01     6.380707010810496e-01
    6.333107568457704e-01     5.387435955592825e-01
    4.779473665564600e-01     4.058422727980679e-01
    3.810187769536677e-01     3.423567923385997e-01
    7.228944981201787e-01     5.970996891140874e-01
    4.412097957053794e-01     7.848185749180456e-01
    2.650977799042060e-01     8.782128750853857e-02
    6.130355014244981e-01     6.001559848449479e-01
    9.254927390811859e-01     5.107653400318706e-01
    6.759555247102171e-01     8.948197698362736e-01
    5.765420868727663e-01     7.771899826055508e-01
    5.047842357889925e-01     3.579561663486917e-01
    6.781998898847558e-01     9.030530568258555e-01
    5.777697885252836e-01     5.500056264081593e-01
    5.423966366891476e-01     7.202389139303629e-01
    6.584742037052129e-01     9.987137878282114e-01
    5.183704018294283e-01     6.769105318559131e-01
    8.358236993597244e-01     5.018762819670171e-01
    4.102320142615749e-01     4.647295168238676e-01
    6.935575561159939e-01     7.259669042588583e-01
    4.287349372082176e-01     8.680048972101012e-01
    8.069048657173383e-01     4.702317304525379e-01
    3.906668724943819e-01     9.024050069964555e-01
    9.181646789822687e-01     3.368626410313738e-01
    8.893122001064503e-01     5.131145372844721e-01
    9.587344375924678e-01     4.567329959200372e-01
    3.157237759949949e-01     1.466114032260302e-01
    6.882440609700731e-01     4.838710784808545e-01
    3.809883602368180e-01     7.708645311907983e-01
    4.642779486949619e-01     7.792524222181251e-01
    5.809628472315514e-01     4.072384418157434e-01
    8.639790213780655e-01     5.083248719539374e-01
    7.885919785506355e-01     7.102176293783371e-01
    1.361563670014603e-01     6.607579423742234e-01
    7.321935840713433e-01     7.863020629462411e-01
    3.779481216058436e-01     1.841542386072119e-01
    2.750210679934983e-01     4.673698286293837e-01
    4.705828586946308e-01     5.440825679641307e-01
    6.944045835661787e-01     5.307712863161757e-01
    8.840729269449704e-01     3.362245221609262e-01
    2.305046466727348e-01     7.929568495841903e-01
    2.563331718287589e-01     4.868628977847363e-01
    4.233451390696568e-01     7.486625583239942e-01
    7.707360814484894e-01     5.296057478031841e-01
    8.497255078832582e-01     5.481960331920962e-01
    1.069997356306484e-01     7.845101815517290e-01
    6.782040258929336e-01     8.129148250494698e-01
    6.316673994414788e-01     7.887562001024765e-01
    3.260113767082208e-01     5.928235619091351e-01
    4.915446841821263e-01     5.716445869547594e-01
    7.494006369774384e-01     5.512915434512949e-01
    6.544638483737707e-01     6.739124765500586e-01
    3.729843381118733e-01     5.199790782969022e-01
    5.103815423267449e-01     6.115297172951561e-01
    7.437763382530650e-01     6.504952668653050e-01
    8.088109235196469e-01     4.326235568303864e-01
    2.969296891618665e-01     5.309828887164656e-01
    3.994638100838156e-01     4.013744167463569e-01
    7.919522029069939e-01     4.553868058049739e-01
    5.821373262070677e-01     9.564771090705532e-01
    3.151283330100618e-01     1.734966117720489e-01
    7.553720820220985e-01     6.220382328878949e-01
    3.170075916544458e-01     1.885988216947763e-01
    1.936395882938914e-01     7.651175188681795e-01
    7.880913060494130e-01     6.484910512174831e-01
    5.284243242412687e-01     4.694122998069370e-01
    5.847389105894643e-01     7.076481661534271e-01
    7.048827857280251e-01     9.743965702919417e-01
    8.154956132186003e-01     5.108246421889072e-01
    3.810059715664854e-01     5.594792970900682e-01
    7.320400793638808e-01     6.165924183668309e-01
    6.549394694751685e-01     5.443211588658621e-01
    5.051671436107611e-01     3.300791711750900e-01
    4.595173420450618e-01     4.899713195190697e-01
    7.863331049850575e-01     3.309539058957398e-01
    4.165816994920232e-01     5.206472231539255e-01
    8.414167438881526e-01     3.087265959774170e-01
    3.094501915821568e-01     5.167638129140043e-01
    8.045067868009458e-01     4.323531579946678e-01
    8.047866300012415e-01     6.544312590425986e-01
    4.121995622514526e-01     8.546859026549606e-01
    8.117087030885659e-01     3.260445483077701e-01
    6.969884183196745e-01     7.819630438329918e-01
    7.364657500836844e-01     8.283668207230489e-01
    3.491399884377864e-01     3.432844874534413e-01
    2.018516769328971e-01     7.519310789226532e-01
    4.781644985903543e-01     6.804989731057745e-01
    6.916429249602184e-01     6.970366288210966e-01
    5.081432103031083e-01     8.701200357630543e-01
    6.487471196541120e-01     9.582387585548154e-01
    3.383121549002501e-01     7.162004949808212e-01
    4.825235103403188e-01     7.045709674641077e-01
    7.512262347506216e-01     4.561946291631686e-01
    4.728001859487191e-01     9.656730664150937e-01
    6.286715517313162e-01     4.739832164693493e-01
    4.505764795091680e-01     2.341534693397221e-01
    8.026814942914753e-01     5.085064260523399e-01
    7.586848550831377e-01     5.197022104641946e-02
    8.876497620104140e-01     3.992664722446754e-01
    9.462831317009135e-01     4.051054163462685e-01
    4.508889591588071e-01     3.104047972272030e-01
    2.698252606777530e-01     7.762163973912941e-01
    2.396388025835677e-01     5.128029290691761e-01
    7.612135241061825e-01     5.607250400152466e-01
    7.751774870520740e-01     4.281316255052389e-01
    5.543982025532705e-01     9.771220648814034e-01
    3.123274203783535e-01     8.065125688391314e-01
    2.090359260648873e-01     6.916575510067445e-01
    7.209943603287432e-01     6.890873569596779e-01
    9.626393627686056e-01     4.572825976045199e-01
    4.396509393311138e-01     2.052145972047658e-01
    4.440995532856670e-01     6.481571551312880e-01
    3.878499022078502e-01     2.591740799630304e-01
    8.414944517191381e-02     7.900199945501376e-01
    2.046901044055589e-01     5.685936152441867e-01
    8.501252069010324e-01     3.735458311051957e-01
    4.820395676915673e-01     7.416493738411881e-01
    1.343264287264045e-01     7.515917826018302e-01
    8.110450364353027e-01     3.291633973449833e-01
    6.379275051052236e-01     8.773937868300954e-01
    5.954183693407438e-01     7.738366291962124e-01
    7.743342068243022e-01     6.976701893975575e-01
    7.493183489992193e-01     6.976278891583469e-01
    1.977050510656216e-01     7.316704765111005e-01
    7.502105366711278e-01     5.226131815673346e-01
    8.081114811413348e-02     7.873096109876908e-01
    3.162742813137025e-01     2.966919667023217e-01
    5.431207460238825e-01     8.202290738863105e-01
    8.056517600208083e-01     4.203068507100660e-01
    9.130804576393554e-01     5.368184596480453e-01
    4.517084740082403e-01     6.353077692297943e-01
    6.358258443837874e-01     7.003014076399965e-01
    4.123270571131327e-01     4.657476509620004e-01
    6.061785854757026e-01     9.167316862057568e-01
    1.119564164145734e-01     6.771774731250656e-01
    5.334524694188598e-01     5.290121879585014e-01
    7.312576353229431e-01     4.961083281620331e-01
    7.735578108986358e-01     6.460394064940270e-01
    7.527967298837722e-01     6.635246221386160e-02
    3.704909242594844e-01     8.711652666127719e-01
    4.744300667937322e-01     8.679455994413842e-01
    3.999549920869281e-01     3.806163576438941e-01
    4.145375199654524e-01     2.888560987767809e-01
    7.651687425873220e-01     3.564102952836983e-01
    7.923435534768685e-01     5.173427805136060e-01
    5.852465503057313e-01     7.538097350559860e-01
    2.567153187342818e-01     6.633640671883897e-01
    8.292443620799721e-01     2.742272288759335e-01
    5.826954094182422e-01     7.870617590829525e-01
    5.228294363545990e-01     5.932328132968520e-01
    2.600937877153079e-01     1.084398502047662e-01
    2.122806503543845e-01     6.739670252183591e-01
    9.368570016240242e-01     4.056941524165597e-01
    3.589161994239345e-01     2.363985525368145e-01
    3.653484519153708e-01     4.335952567151098e-01
    8.812827028936964e-01     5.662905086143134e-01
    9.497426484881268e-01     4.381648053006421e-01
    3.446772646233592e-01     2.854607085203625e-01
    3.813226125449700e-01     4.934589183662493e-01
    3.233777748099714e-01     7.370791842075902e-01
    4.328786047403355e-01     9.334636869361871e-01
    5.185810411866074e-01     7.476123823040318e-01
    2.319156392351337e-01     6.265374039376141e-01
    2.545942955811192e-01     8.181488100024357e-01
    5.343079829901058e-01     9.997660868061986e-01
    8.184282361138928e-01     4.336217519583326e-01
    3.951530610571293e-01     6.265194512248117e-01
    9.615542814712054e-01     5.192827219019351e-01
    4.228916539713554e-01     6.447049029667141e-01
    7.895134646619272e-01     3.958893862677690e-01
    3.561197936516991e-01     4.925179584215834e-01
    5.350910998137808e-01     6.820218788418089e-01
    5.635843485372194e-01     3.337838182057401e-01
    5.624488085294386e-01     8.505894292037984e-01
    7.333045787117132e-01     5.796838658582716e-01
    5.581363229852792e-01     5.042784538690034e-01
    6.152504662742481e-01     8.438226379251996e-01
    2.980891983011253e-01     8.530439108142562e-01
    4.621632407561006e-01     6.755594965268388e-01
    8.202581334927418e-01     3.223084700474526e-01
    2.696339352632765e-01     8.094847079477500e-01
    7.359472568945143e-01     6.775683262358867e-01
    8.236003175127563e-01     4.856212046607140e-01
    6.380974265167598e-01     7.876492846559557e-01
    7.300491568435332e-01     9.603794155362884e-01
    2.408333674673333e-01     6.013993792805660e-01
    3.864600632413677e-01     4.740119431939579e-01
    3.543741009867712e-01     2.070547588001611e-01
    4.851776898125396e-01     6.674653333683790e-01
    7.262258204025612e-01     9.536941096342021e-01
    4.657292905851140e-01     9.335170611253593e-01
    8.720973045403133e-01     5.299316220069884e-01
    7.771811175865265e-01     4.320058499950290e-01
    3.052573920540360e-01     9.756006250925831e-02
    5.271960985089307e-01     8.886428621504555e-01
    7.555082351140221e-01     3.380272887859663e-01
    4.900936204237619e-01     4.220555773550488e-01
    4.001305810345255e-01     2.893567710909714e-01
    6.770185539356613e-01     7.064640667004678e-01
    6.420770224698510e-01     6.710354148633911e-01
    2.781516098848896e-01     7.075810010087431e-01
    2.397764040414320e-01     8.304823620456208e-01
    5.413751948638161e-01     4.619405176200937e-01
    3.479571635275389e-01     1.407969112643925e-01
    7.603173799933194e-01     1.719185523301160e-01
    6.828637487351628e-01     9.121382519676613e-01
    1.049302472659634e-01     7.700314691208984e-01
    6.341404106173396e-01     4.427569191052136e-01
    3.860073861843156e-01     5.364104818764147e-01
    5.052980511294171e-01     4.054200779207439e-01
    7.992560611154386e-01     4.181886724049072e-01
    3.064624185328579e-01     3.786942358220003e-01
    4.922568131145213e-01     9.784957739614275e-01
    3.009617525673672e-01     7.067869544030791e-02
    4.624592546449993e-01     8.139771174498265e-01
    3.831877792445455e-01     9.336939907532186e-01
    7.397886071480959e-01     7.012946453104472e-01
    5.762785235707245e-01     9.784109895023407e-01
    3.157105043156487e-01     8.456496186731042e-01
    9.260606559891851e-01     5.283579382930623e-01
    7.506766153692943e-01     1.592842245144388e-01
    7.222607670791152e-01     6.662776385045780e-01
    1.079837848889563e-01     6.960435058320931e-01
    7.982098977569492e-01     5.094802131175968e-01
    6.557586797079160e-01     5.610682873977147e-01
    3.497376212002156e-01     1.959688051389736e-01
    8.312176026344277e-01     2.635643439619698e-01
    3.506768389864395e-01     6.681190164810136e-01
    3.172133011807888e-01     6.783402024397596e-01
    4.773112449338087e-01     8.826328720641164e-01
    9.318102094603588e-01     4.878292337648974e-01
    4.745682634110414e-01     7.278914946573380e-01
    4.180850572593323e-01     8.781048485232349e-01
    5.152049722661145e-01     5.233175455686767e-01
    5.408872162219409e-01     8.917869103251820e-01
    2.012589530358767e-01     6.959478449522744e-01
    3.593131342496769e-01     1.879484357507044e-01
    6.231181872636241e-01     6.159295245560206e-01
    6.112760469465025e-01     8.932815671045361e-01
    7.569205707372940e-01     3.733718861516789e-01
    6.854298861970806e-01     4.800976367434608e-01
    8.334207149588020e-01     4.420188316046364e-01
    4.546288351146194e-01     5.721440117855781e-01
    4.744790765629927e-01     2.863746229123451e-01
    6.694629100824974e-01     8.197901264723021e-01
    1.687183745200476e-01     8.117521663644032e-01
    7.169876932860658e-01     6.657330802733286e-01
    5.206838299583970e-01     9.360006762912947e-01
    8.110129119189484e-01     1.959664883241016e-01
    8.401059006964082e-01     2.945498667538746e-01
    6.466241907215778e-01     7.831044218039485e-01
    5.960886955811128e-01     3.883013934425061e-01
    5.593083056434704e-01     4.134305345696926e-01
    3.307301125153387e-01     1.363867645475041e-01
    7.290238170769284e-01     5.637749713549488e-01
    7.306310627904349e-01     9.280218969842051e-01
    5.094782989460002e-01     5.653865900663645e-01
    1.675793845841210e-01     8.080480719211364e-01
    4.824757402922653e-01     2.504155744762310e-01
    9.109846917628743e-01     5.027652984263965e-01
    8.681941309835962e-01     5.150241761609067e-01
    6.277019698146504e-01     7.175718120960664e-01
    4.880868249786272e-01     8.632495528558676e-01
    2.937185524948156e-01     8.683947475068351e-01
    6.452373885228173e-01     7.983839208172723e-01
    7.061664973715798e-01     9.886696936237077e-01
    2.901595620364734e-01     4.043560919061621e-01
    4.411750109018937e-01     5.368203818619922e-01
    3.815062845113733e-01     5.808382044885526e-01
    6.322726057139030e-01     9.591928309542856e-01
    5.345154045134948e-01     3.438680641652501e-01
    1.180348115556191e-01     6.823604623156377e-01
    2.826688637658407e-01     8.017015645781853e-01
    3.955478123166016e-01     3.901304232575622e-01
    5.251617523592924e-01     4.642258518925537e-01
    3.763076085257450e-01     2.143290144535881e-01
    5.281692487114625e-01     6.778287740628134e-01
    1.545644167593104e-01     7.760668814980752e-01
    3.251245811314527e-01     5.538114729156015e-01
    5.088211726901762e-01     7.221686974774786e-01
    3.470792728331124e-01     3.522768443284615e-01
    5.746200367584844e-01     8.071413766731490e-01
    3.649073468155313e-01     6.818336888537180e-01
    8.066510135911122e-01     5.934484460020265e-01
    8.729439550151897e-01     5.827640817290448e-01
    6.998745242792589e-01     6.900041991519850e-01
    5.593611068070272e-01     8.379687492331422e-01
    3.142047041844641e-01     4.305093376063373e-01
    5.987629524692750e-01     3.701140755567913e-01
    3.241586356323222e-01     3.400953912585037e-01
    5.888931933143151e-01     3.978488735081313e-01
    7.322976960087273e-01     8.363570118047032e-01
    8.165150507932137e-01     5.844206661096107e-01
    2.517907950224213e-01     7.363997882576954e-01
    5.787697472115600e-01     8.940706029369621e-01
    5.880471996704569e-01     4.503453864724216e-01
    8.838037923343920e-01     5.056521013144464e-01
    5.244954007084809e-01     3.649962656533782e-01
    5.319150143743860e-01     4.518820566070941e-01
    6.290658806465567e-01     4.363411495663061e-01
    3.172972946640744e-01     4.423136876498468e-01
    3.063331467012034e-01     7.454743171896843e-01
    3.753231150894279e-01     4.512007723192573e-01
    7.064366757331186e-01     8.996745280400587e-01
    6.402385861157591e-01     4.847717502761957e-01
    7.130199491224413e-01     9.786750622453756e-01
    5.044487784392049e-01     5.919250108016405e-01
    6.559563722012610e-01     8.353298947747795e-01
    4.410540732409572e-01     2.305309595096362e-01
    3.546068018284007e-01     1.164666153335843e-01
    5.654275489901402e-01     6.994932796452040e-01
    8.530103283650469e-01     6.061488884366963e-01
    8.107549569349246e-01     6.578664019332294e-01
    5.428486085341341e-01     6.469766023254095e-01
    4.664283003805281e-02     7.215112043830675e-01
    6.125524016591805e-01     8.164006200357835e-01
    6.398354825107514e-01     9.948216950740656e-01
    1.635945646826887e-01     7.319813158245664e-01
    3.735175087213327e-01     1.377854675377692e-01
    4.550739174213351e-01     9.010247893858206e-01
    6.587797800254849e-01     5.037041802586834e-01
    4.081897053347316e-01     1.602980368722636e-01
    5.477947081681500e-01     8.681538896240097e-01
    9.832608173524919e-01     4.756544061393222e-01
    6.722973403356868e-01     5.116554695015160e-01
    5.694340903792741e-01     3.880862523419332e-01
    7.049319625706780e-01     9.697064797075204e-01
    3.881216831630209e-01     4.926839983167443e-01
    5.764245481164511e-01     5.169501057533458e-01
    4.162027314322470e-01     5.115187122267776e-01
    3.531670489574779e-01     6.648664572368569e-01
    6.630405217401149e-01     4.718861777869159e-01
    3.494226068846431e-01     5.928794589265853e-01
    2.952059506820054e-01     8.196343041644709e-01
    8.139951210400103e-01     5.062964994450402e-01
    7.325198444091902e-01     7.208702591791697e-01
    9.036696117718112e-01     3.413525573060320e-01
    6.981165319115603e-01     6.941446597683979e-01
    7.512404049423681e-01     6.851081966443051e-01
    3.354576629724730e-01     3.055660314828803e-01
    5.496214454877616e-01     8.704582554579521e-01
    7.126143711997520e-01     7.788007492023733e-01
    5.972324908681228e-01     6.480509037917048e-01
    4.136178020034730e-01     3.248588881323535e-01
    3.696828109066557e-01     5.675761728525366e-01
    4.556854930615519e-01     2.341519348307362e-01
    6.071497171479910e-01     5.486444839222220e-01
    7.357393832779116e-01     5.084775574103433e-01
    7.353068608056879e-01     6.661606837536617e-01
    2.324475622834644e-01     8.441630883775655e-01
    2.972914236252809e-01     3.784401628162145e-01
    3.233748897376378e-01     3.448762801238993e-01
    8.251493277244232e-01     3.027142888220354e-01
    6.967698140687633e-01     5.549691327350105e-01
    2.643171582456760e-01     7.920549273160559e-01
    2.606295582598375e-01     7.476830769692011e-02
    4.790889413227859e-01     4.481097569298980e-01
    7.537444634623268e-01     7.343888855413264e-01
    8.562426082391694e-01     2.333973393057563e-01
    1.553080277714675e-01     8.151887629313794e-01
    3.887370509634591e-01     5.132779353816370e-01
    7.678599971269895e-01     7.033402543888611e-01
    8.011829797479957e-01     6.952319049805442e-01
    4.299356875730561e-01     6.367351436329894e-01
    6.479549517895221e-01     8.897133585663544e-01
    7.108985844714264e-01     4.748547900880837e-01
    3.330630443789999e-01     1.201016994500733e-01
    4.830799638255078e-01     3.616864128118622e-01
    5.849221248450404e-01     4.930262987346536e-01
    7.697892798549989e-01     7.235108565766646e-01
    3.080185003691174e-01     2.421023122307424e-01
    8.772855153910277e-01     4.936890111053560e-01
    5.395250042834033e-01     5.331591026415246e-01
    6.211925820870782e-01     4.825777731510724e-01
    8.570150708404803e-01     2.335309161426906e-01
    9.151769128014733e-01     4.430428630757208e-01
    6.169350336756171e-01     6.265900842875454e-01
    4.642528987028146e-01     7.330450749080277e-01
    5.207347975354506e-01     8.399646768984921e-01
    6.565198052655358e-01     6.323857943075570e-01
    5.753308284944186e-01     9.642207888888285e-01
    5.265418387580278e-01     7.561123217813533e-01
    8.600851681850362e-01     3.406921774978888e-01
    6.053391359075003e-01     8.398088919988353e-01
    5.431000516404858e-01     6.731308320774385e-01
    2.983754192814185e-01     5.537016995136967e-01
    3.849399094491383e-01     6.032859290379305e-01
    9.338215399203323e-01     5.197998713815749e-01
    3.539013966482345e-01     8.583551429464669e-01
    3.856021495636893e-01     8.763657951005280e-01
    3.597132713757720e-01     5.261278511895450e-01
    2.316623086060000e-01     7.673145805667635e-01
    6.319569905390200e-01     9.544789222936700e-01
    2.895890183486955e-01     5.892389659463158e-01
    2.804588785710331e-01     8.809454444176978e-02
    8.338677696870643e-01     4.075365812581925e-01
    9.564166474008349e-01     5.063542343024486e-01
    3.926967435107870e-01     7.691069566416822e-01
    4.667765051707300e-01     4.626190445552145e-01
    1.629901982382530e-01     6.721689139529033e-01
    8.667678746096182e-01     3.553064092572292e-01
    4.207125685871279e-01     5.010276861922381e-01
    3.127683242194568e-01     1.387266952694735e-01
    1.267882159594866e-01     6.655539542646972e-01
    2.736689840000474e-01     7.778552278036581e-01
    1.985148716363854e-01     6.034188233657429e-01
    3.269342716122250e-01     8.968682315497610e-01
    5.479441606544565e-01     6.927117883198757e-01
    7.026695122722820e-01     6.164209404646896e-01
    2.805293163207836e-01     7.502357526235329e-01
    3.572971600576250e-01     8.821190868901433e-01
    1.388402179571411e-01     7.224191949205323e-01
    5.194558913758854e-01     3.957941169622560e-01
    4.770366723401891e-01     5.182161833418205e-01
    6.485460750926623e-01     7.470048284895537e-01
    9.237045885949586e-01     5.087111097715962e-01
    8.429453916791183e-01     3.755153918694673e-01
    7.321076696991906e-01     8.890647471602990e-01
    3.574767525989313e-01     6.407719402566145e-01
    8.322650352732981e-01     3.873597385900640e-01
    8.323427784149520e-01     2.510186084370627e-01
    3.122847790870468e-01     3.107878479088511e-01
    6.319560742293672e-01     6.829076650431695e-01
    2.857785479478455e-01     3.546575081925527e-01
    6.247894309172979e-01     5.539253522747378e-01
    1.167960934117545e-01     7.849379480940605e-01
    5.830979255002965e-01     9.778516773362488e-01
    6.552124288821118e-01     9.199843691560654e-01
    2.279364727566172e-01     7.773848278084270e-01
    4.338911594662513e-01     3.345555395984208e-01
    4.799006388870757e-01     4.475741713379789e-01
    3.212602161022933e-01     3.281145170709142e-01
    8.067615325685437e-01     5.789537635044331e-01
    3.013684359792711e-01     3.940013021133554e-01
    7.723445696447847e-01     6.503957382901528e-01
    7.612838287659998e-01     5.694138389770900e-01
    2.796471136015495e-01     6.234945958707658e-01
    5.111237631841267e-01     7.888307822674170e-01
    7.723641844721294e-01     4.830131935245124e-01
    1.066467045779500e-01     7.181485670054218e-01
    5.019312609438096e-01     5.132300780328785e-01
    2.475605409115796e-01     6.769567671724857e-01
    5.808952546656554e-01     8.991294259349046e-01
    3.913797477206170e-01     7.255323152903607e-01
    9.096034100648506e-01     3.266176533769374e-01
    1.748580935922004e-01     6.996265901268636e-01
    3.358533692397840e-01     3.698732369967695e-01
    8.192652602979482e-01     1.631768309598877e-01
    4.042470524836463e-01     2.095431798743912e-01
    8.877869870916264e-01     5.673656301814991e-01
    8.632828718347826e-01     2.288038427511879e-01
    5.730921119548101e-01     6.109705738425601e-01
    2.433428385730843e-01     7.697640284806250e-01
    8.271007645067436e-01     3.695796329787947e-01
    7.810347861586268e-01     7.681460647032534e-02
    5.884700920066718e-01     6.406802344011965e-01
    4.668517123800296e-01     2.258121749799121e-01
    1.913622320751501e-01     7.857534738387938e-01
    7.975154553260245e-01     4.301164562152120e-01
    5.480254131139780e-01     6.475667007815521e-01
    1.176811224053472e-01     7.703987192541304e-01
    4.672913440360013e-01     7.219252541171862e-01
    4.626444390533232e-01     3.043047523166013e-01
    3.552305005579962e-01     1.371948987535324e-01
    3.834598649909104e-01     7.784337264222322e-01
    2.674412166788348e-01     1.307662745456214e-01
    3.525091112807665e-01     5.806117676949606e-01
    7.541918114029317e-01     1.846933323942678e-01
    1.329340697601944e-01     7.436845548402420e-01
    4.399838962601949e-01     6.934251357401189e-01
    1.237381089949523e-01     6.863257987836974e-01
    3.043571634732285e-01     7.262757904600978e-01
    6.601399746922313e-01     6.384923039332957e-01
    8.272123289178397e-01     4.901016983056018e-01
    7.693930683782949e-01     2.297553083474698e-01
    8.218140607508584e-01     2.148798534302917e-01
    6.635092462215211e-01     6.396465174957437e-01
    4.705651839187122e-01     7.280670452275005e-01
    7.657970193474436e-01     6.488750799078865e-01
    4.706428858667183e-01     5.190576446048524e-01
    8.702206904713982e-01     6.012277976536858e-01
    4.686429773832146e-02     7.390942970471172e-01
    5.112415872043639e-01     3.217269282462174e-01
    2.916564627036222e-01     8.353521745016808e-01
    4.027418685806539e-01     3.280667415827623e-01
    5.871418351192519e-01     3.825983464372856e-01
    8.338785701608288e-01     6.100144010755113e-01
    3.285273504335517e-01     3.943175791340434e-01
    3.452252126810924e-01     6.934651584685306e-01
    5.140286946791155e-01     7.687670177788443e-01
    6.461605890992311e-01     4.830469959918207e-01
    2.076583074636922e-01     6.955772569612783e-01
    2.692228238394747e-01     2.347469178055717e-02
    3.556339942126219e-01     6.644436972241534e-01
    2.521723193877091e-01     1.476476568738029e-01
    3.939743268653066e-01     2.010471713993118e-01
    6.921397428176946e-01     5.416558316964428e-01
    6.507031367900898e-01     5.552932044405730e-01
    4.309936536161463e-01     2.571823329163491e-01
    7.567161690629161e-01     5.127713533119714e-01
    5.763200722546762e-01     9.967852136183872e-01
    5.058742650100925e-01     8.101482894512637e-01
    2.664692145541030e-01     6.421111146488176e-01
    1.600009661165490e-01     7.913414105895109e-01
    6.824955062807732e-01     5.456535678299173e-01
    6.688769638437649e-01     5.048625222293377e-01
    3.446026815200598e-01     1.273176743956251e-01
    1.955642172701635e-01     7.058295857008964e-01
    1.869440827876141e-01     7.993069553391474e-01
    7.408793371289215e-01     6.621563060270852e-01
    5.356397631963764e-01     4.456068405461922e-01
    3.310948272397283e-01     6.454369537992791e-01
    2.039510861471391e-01     6.218579636735283e-01
    4.682465041125137e-01     9.523403987336820e-01
    2.540480767581246e-01     6.544188374225335e-01
    2.178992346899832e-01     7.224956264827018e-01
    3.380438394068316e-01     5.106340267739087e-01
    3.712640765179724e-01     4.049241632667375e-01
    2.973142436272671e-01     7.203667135628861e-01
    3.718092343989687e-01     2.229065913584237e-01
    4.215120653625929e-01     3.451445635050634e-01
    6.272506645247803e-01     8.305922652288061e-01
    1.263670204595992e-01     7.332553833573970e-01
    2.852275765987495e-01     3.014600957766453e-01
    5.504012334029481e-01     9.845908058938847e-01
    7.440163733816610e-01     8.582298665263179e-01
    4.391546099687982e-01     8.039764274917186e-01
    7.311713911241640e-01     9.612540164876240e-01
    5.425650737244392e-01     4.764382296611989e-01
    7.472902741428588e-01     5.782350266983730e-01
    5.512701831576434e-01     6.238648466534930e-01
    3.989839269878286e-01     8.450898981722832e-01
    1.671524276907903e-01     8.303300870166479e-01
    3.772710723049763e-01     3.976950154341161e-01
    4.489352367592154e-01     7.762933007306972e-01
    1.465698186850317e-02     7.553520951751087e-01
    3.898390154476303e-01     3.978118556054245e-01
    6.461189635701751e-01     8.635310665307715e-01
    1.822310668469624e-01     6.289699376857727e-01
    5.699561683810144e-01     6.847706809546554e-01
    2.529092112795107e-01     3.654597265712568e-01
    6.474855639400388e-01     4.769553713852916e-01
    4.676111160365860e-01     6.870920613003112e-01
    3.590872338976485e-01     8.820915058221150e-01
    3.592648543236792e-01     2.989526574452838e-01
    2.308998255189153e-01     5.420655702102231e-01
    7.574490269962160e-01     1.378386378656656e-01
    8.134895875617963e-01     1.718475388980620e-01
    6.149794058818304e-01     5.003927599132796e-01
    2.444300911417945e-01     7.606321503949095e-01
    7.523943052654123e-01     1.001839717278659e-01
    4.299108430938251e-01     5.139640133759034e-01
    3.444213775979187e-01     3.003409560335750e-01
    2.235502221694412e-01     7.690135862978402e-01
    6.808116865940020e-01     7.621826544870686e-01
    6.131496971483260e-01     8.739968017417858e-01
    2.758886529702593e-01     4.408961010718805e-01
    3.924088245934292e-01     8.453014773159319e-01
    5.569744388158374e-01     9.993493808275196e-01
    5.503518964542800e-01     9.708219395695472e-01
    2.329073423971773e-01     7.978095661504699e-01
    5.320717095698895e-01     9.917936837152221e-01
    7.523675708766625e-01     2.791677336849860e-01
    4.018222668888980e-01     1.953823214513032e-01
    6.047038781811681e-01     5.173774062074299e-01
    5.758893006043517e-01     6.160988176457440e-01
    6.654938999371246e-01     8.154856049339479e-01
    7.673435122831670e-01     1.387914318260673e-01
    4.619810183022979e-01     8.186778134884114e-01
    4.286302718134219e-01     6.081409314792672e-01
    6.649213226976135e-01     9.459537495510554e-01
    8.087341346406098e-01     1.293859126232934e-01
    9.695887124225735e-01     4.661430473004970e-01
    2.937528113660579e-01     1.748112805741519e-01
    7.641081428694041e-01     4.110343973708397e-02
    5.537722801753060e-01     4.490008012626211e-01
    2.708738255419891e-01     8.232379127401538e-01
    4.575633191695425e-01     7.442472084073006e-01
    2.534913029365902e-01     4.441724305443120e-01
    3.575802529786696e-01     8.533563023798625e-01
    6.052407869289185e-01     6.597682068681501e-01
    6.502862457594667e-01     5.515067150102951e-01
    5.205358306943961e-01     3.882327689248632e-01
    5.643987339426459e-01     6.921456988220545e-01
    6.229817874211022e-01     6.679284362325560e-01
    7.241035716061262e-01     9.284936960037512e-01
    4.159352835500008e-01     2.168608129434834e-01
    9.547424144448633e-01     5.159708654437239e-01
    2.153845262360045e-01     7.731727533530782e-01
    5.602628159786484e-01     9.368778640727896e-01
    3.175951226844361e-01     4.122950633688367e-01
    3.944445399172541e-01     2.884594448174568e-01
    6.536410170309304e-01     7.647659879744557e-01
    6.888670148208323e-01     9.328738113314500e-01
    5.011025825055787e-01     5.064315969373370e-01
    2.426600477261590e-01     6.241190425784193e-01
    6.980876145308372e-01     7.172407333577283e-01
    3.779135240812789e-01     4.169965908433098e-01
    4.472769766301217e-01     7.687429763532451e-01
    7.397459711408204e-01     6.710018814397050e-01
    5.414101701098285e-01     6.732710500050961e-01
    6.370668180907425e-01     5.936042119888616e-01
    5.284568369702483e-01     7.696456741771287e-01
    1.491706618259913e-01     7.309452023126620e-01
    5.811159152469733e-01     6.372094768051869e-01
    9.216276210209043e-01     3.875366394132075e-01
    7.379473797633470e-01     7.670738186689661e-01
    3.614194862806757e-01     4.639919002739243e-01
    5.115516219738064e-01     3.090741941133390e-01
    8.691218965405564e-01     3.403514821272036e-01
    4.062398130945719e-01     2.922284374614470e-01
    5.294481756918076e-01     4.946394922731309e-01
    3.808697803704649e-01     7.587876556886796e-01
    4.220667346152273e-01     8.036652603649822e-01
    2.953666912132940e-01     6.439008160512970e-01
    8.881352785113922e-01     4.748705544102786e-01
    7.094622717389689e-01     5.261334684124749e-01
    7.890780243043113e-01     3.647954799147204e-01
    6.057931877826821e-01     9.172171690902235e-01
    6.820179477749165e-01     9.003861290776823e-01
    3.132964239362638e-01     8.392400055598020e-01
    2.175091098454316e-01     7.310561354459970e-01
    6.179212249789660e-01     8.592108995326830e-01
    8.404734154917239e-01     4.259686673041143e-01
    1.475139377164032e-01     7.281641852982215e-01
    8.469550176715596e-01     2.462676202252365e-01
    4.971446348580918e-01     4.778997817022778e-01
    1.270634258404127e-01     7.331930566785537e-01
    8.202392354074662e-01     2.407322609493107e-01
    1.102113579848222e-01     7.454027085453877e-01
    2.639007138676814e-01     7.978979422922117e-02
    3.601260156623125e-01     4.607850748551415e-01
    6.551641417669445e-01     6.535426184853593e-01
    4.051185364118985e-01     6.041672928911200e-01
    9.298789385167763e-01     3.682038521355562e-01
    2.523062293327704e-01     2.977697792521428e-02
    5.887804701951361e-01     4.293858161258353e-01
    5.161114444032442e-01     9.011965010843095e-01
    1.716673376894088e-01     6.986537751700345e-01
    3.271340377069487e-02     7.536278709316269e-01
    3.224120246183383e-01     4.324628399395775e-01
    3.613449235850192e-01     2.233359232458151e-01
    2.687479298144161e-01     8.140675440601188e-01
    4.836233744692903e-01     3.976406082349104e-01
    6.547904699645354e-01     4.298446751294326e-01
    5.486026206413542e-01     6.386892365873051e-01
    5.472522074286638e-01     7.702841521459511e-01
    2.592282661563541e-01     6.485112516820527e-01
    7.975761643599510e-01     2.453930303813635e-01
    7.073575907764208e-01     8.231926240036954e-01
    6.404442392781329e-01     5.123635865309967e-01
    1.828831125826800e-01     8.239461905153269e-01
    2.427981424092913e-01     8.484532725803555e-01
    6.235755152962333e-01     8.886395006308395e-01
    5.388164097378347e-01     6.891669571207762e-01
    7.753735978936481e-01     6.798570469619543e-01
    2.233052752921576e-01     7.433785633427297e-01
    7.446271842831396e-01     9.927995204974644e-01
    2.469714507658984e-01     6.147946493835381e-01
    3.479310025484750e-01     5.623129733351537e-01
    4.040590733516372e-01     7.886988710212003e-01
    7.660099922976046e-01     7.247291095526197e-01
    7.856487378791875e-01     4.892368919368847e-01
    2.946621779444045e-01     8.214231942229945e-01
    3.162586204853721e-01     2.460058417979706e-01
    7.277745262714893e-01     9.335636280698399e-01
    4.375418002756928e-01     4.956079743176246e-01
    1.450390872061231e-01     6.854386189435465e-01
    5.162597924899676e-01     8.502623843519708e-01
    3.129104013192521e-01     4.202711737426865e-01
    2.032955734253883e-02     7.531523777967161e-01
    6.366323938381562e-01     4.934886005550588e-01
    7.139879789562258e-01     6.927703202158403e-01
    3.051761001053995e-01     8.074895627086006e-01
    9.163374185816492e-01     4.998254908466465e-01
    5.375952172593309e-01     5.359320866730193e-01
    5.357214393461021e-01     4.370377588945480e-01];

if Npts <= 4998
    centers=pts(1:Npts,:);
else
    error('Too many points are requested by the user');
end