
function xyzw=cub_sphere_low(deg)

%--------------------------------------------------------------------------
% Object:
%--------------------------------------------------------------------------
% Cubature over the unit sphere.
% This routine loads spherical designs computed by Rob Womersley, useful
% for cubature on the unit 2-sphere, with algebraic degree of precision at
% most "deg".
%
% They are named by R.S. Womersley
%       "Efficient Spherical Designs with Good Geometric Properties"
% and are available at 
%        https://web.maths.unsw.edu.au/~rsw/Sphere/EffSphDes/ss.html
%
% This procedure helps using the right formula for the right degree.
%--------------------------------------------------------------------------
% Input:
%--------------------------------------------------------------------------
% deg: algebraic degree of precision of the cubature rule.
%--------------------------------------------------------------------------
% Output:
%--------------------------------------------------------------------------
% xyzw: nodes and weights are stored in a "M x 4" matrix, where "M" is the
%     cardinality of the rule.
%     The k-th node, in cartesian coordinates correspond to "xyzw(k,1:3)",
%     while the k-th weight is "xyzw(k,4)".
%     The weights are choosen so that their sum is "4*pi"
%--------------------------------------------------------------------------
% Reference:
%--------------------------------------------------------------------------
% All these rules are taken from
% "Efficient Spherical Designs with Good Geometric Properties"
% and are studied and stored by Rob Womersley.
%--------------------------------------------------------------------------
% Rules cardinalities:
%--------------------------------------------------------------------------
%  	 |   1 |     3 |     0 | 8.882e-16 |
%  	 |   2 |     6 |     0 | 4.441e-16 |
%  	 |   3 |     8 |     0 | 1.277e-15 |
%  	 |   4 |    14 |     0 | 1.332e-15 |
%  	 |   5 |    18 |     0 | 2.044e-14 |
%  	 |   6 |    26 |     0 | 1.277e-15 |
%  	 |   7 |    32 |     0 | 1.610e-15 |
%  	 |   8 |    42 |     0 | 2.220e-15 |
%  	 |   9 |    50 |     0 | 1.756e-15 |
%  	 |  10 |    62 |     0 | 1.721e-15 |
%  	 |  11 |    72 |     0 | 2.373e-15 |
%  	 |  12 |    86 |     0 | 1.832e-15 |
%  	 |  13 |    98 |     0 | 3.608e-15 |
%  	 |  14 |   114 |     0 | 4.441e-15 |
%  	 |  15 |   128 |     0 | 3.858e-15 |
%  	 |  16 |   146 |     0 | 3.011e-15 |
%  	 |  17 |   163 |     0 | 4.441e-15 |
%  	 |  18 |   182 |     0 | 3.515e-15 |
%  	 |  19 |   201 |     0 | 7.105e-15 |
%  	 |  20 |   222 |     0 | 6.217e-15 |
%  	 |  21 |   243 |     0 | 4.441e-15 |
%  	 |  22 |   266 |     0 | 7.105e-15 |
%  	 |  23 |   289 |     0 | 6.217e-15 |
%  	 |  24 |   314 |     0 | 3.997e-15 |
%  	 |  25 |   339 |     0 | 1.243e-14 |
%  	 |  26 |   366 |     0 | 9.770e-15 |
%  	 |  27 |   393 |     0 | 6.217e-15 |
%  	 |  28 |   422 |     0 | 9.770e-15 |
%  	 |  29 |   451 |     0 | 2.043e-14 |
%  	 |  30 |   482 |     0 | 2.132e-14 |
%  	 |  31 |   513 |     0 | 1.243e-14 |
%  	 |  32 |   546 |     0 | 1.865e-14 |
%  	 |  33 |   579 |     0 | 7.105e-15 |
%  	 |  34 |   614 |     0 | 9.770e-15 |
%  	 |  35 |   649 |     0 | 2.576e-14 |
%  	 |  36 |   686 |     0 | 1.732e-14 |
%  	 |  37 |   723 |     0 | 3.782e-15 |
%  	 |  38 |   762 |     0 | 1.465e-14 |
%  	 |  39 |   801 |     0 | 1.954e-14 |
%  	 |  40 |   842 |     0 | 2.354e-14 |
%  	 |  41 |   883 |     0 | 2.665e-14 |
%  	 |  42 |   926 |     0 | 2.220e-14 |
%  	 |  43 |   969 |     0 | 1.599e-14 |
%  	 |  44 |  1014 |     0 | 2.132e-14 |
%  	 |  45 |  1059 |     0 | 3.375e-14 |
%  	 |  46 |  1106 |     0 | 3.553e-14 |
%  	 |  47 |  1153 |     0 | 2.398e-14 |
%  	 |  48 |  1202 |     0 | 1.510e-14 |
%  	 |  49 |  1251 |     0 | 2.665e-14 |
%  	 |  50 |  1302 |     0 | 1.910e-14 |
%  	 |  51 |  1353 |     0 | 3.153e-14 |
%  	 |  52 |  1406 |     0 | 1.865e-14 |
%  	 |  53 |  1459 |     0 | 2.309e-14 |
%  	 |  54 |  1514 |     0 | 6.306e-14 |
%  	 |  55 |  1569 |     0 | 2.975e-14 |
%  	 |  56 |  1626 |     0 | 2.753e-14 |
%  	 |  57 |  1683 |     0 | 5.196e-14 |
%  	 |  58 |  1742 |     0 | 3.109e-14 |
%  	 |  59 |  1801 |     0 | 3.597e-14 |
%  	 |  60 |  1862 |     0 | 5.684e-14 |
%  	 |  61 |  1923 |     0 | 5.373e-14 |
%  	 |  62 |  1986 |     0 | 6.306e-14 |
%  	 |  63 |  2049 |     0 | 3.197e-14 |
%  	 |  64 |  2114 |     0 | 3.553e-14 |
%  	 |  65 |  2179 |     0 | 2.309e-14 |
%  	 |  66 |  2246 |     0 | 4.041e-14 |
%  	 |  67 |  2313 |     0 | 4.219e-14 |
%  	 |  68 |  2382 |     0 | 4.302e-15 |
%  	 |  69 |  2451 |     0 | 4.086e-14 |
%  	 |  70 |  2522 |     0 | 9.415e-14 |
%  	 |  71 |  2593 |     0 | 5.684e-14 |
%  	 |  72 |  2666 |     0 | 2.887e-14 |
%  	 |  73 |  2739 |     0 | 7.994e-14 |
%  	 |  74 |  2814 |     0 | 1.008e-13 |
%  	 |  75 |  2889 |     0 | 4.707e-14 |
%  	 |  76 |  2966 |     0 | 2.753e-14 |
%  	 |  77 |  3043 |     0 | 5.063e-14 |
%  	 |  78 |  3122 |     0 | 9.726e-14 |
%  	 |  79 |  3201 |     0 | 8.793e-14 |
%  	 |  80 |  3282 |     0 | 4.130e-14 |
%  	 |  81 |  3363 |     0 | 5.507e-14 |
%  	 |  82 |  3446 |     0 | 3.286e-14 |
%  	 |  83 |  3529 |     0 | 7.239e-14 |
%  	 |  84 |  3614 |     0 | 9.281e-14 |
%  	 |  85 |  3699 |     0 | 7.638e-14 |
%  	 |  86 |  3786 |     0 | 6.528e-14 |
%  	 |  87 |  3873 |     0 | 1.710e-13 |
%  	 |  88 |  3962 |     0 | 7.416e-14 |
%  	 |  89 |  4051 |     0 | 1.226e-13 |
%  	 |  90 |  4142 |     0 | 1.248e-13 |
%  	 |  91 |  4233 |     0 | 1.279e-13 |
%  	 |  92 |  4326 |     0 | 3.819e-14 |
%  	 |  93 |  4419 |     0 | 1.288e-13 |
%  	 |  94 |  4514 |     0 | 5.640e-14 |
%  	 |  95 |  4609 |     0 | 1.270e-13 |
%  	 |  96 |  4706 |     0 | 7.105e-14 |
%  	 |  97 |  4803 |     0 | 1.124e-13 |
%  	 |  98 |  4902 |     0 | 5.507e-14 |
%  	 |  99 |  5001 |     0 | 1.403e-13 |
%  	 | 100 |  5102 |     0 | 6.439e-14 |
%  	 | 101 |  5203 |     0 | 2.407e-13 |
%  	 | 102 |  5306 |     0 | 6.617e-14 |
%  	 | 103 |  5409 |     0 | 9.681e-14 |
%  	 | 104 |  5514 |     0 | 2.300e-13 |
%  	 | 105 |  5619 |     0 | 1.075e-13 |
%  	 | 106 |  5726 |     0 | 1.563e-13 |
%  	 | 107 |  5833 |     0 | 2.114e-13 |
%  	 | 108 |  5942 |     0 | 1.874e-13 |
%  	 | 109 |  6051 |     0 | 1.878e-13 |
%  	 | 110 |  6162 |     0 | 7.372e-14 |
%  	 | 111 |  6273 |     0 | 1.110e-13 |
%  	 | 112 |  6386 |     0 | 6.173e-14 |
%  	 | 113 |  6499 |     0 | 4.097e-15 |
%  	 | 114 |  6614 |     0 | 1.825e-13 |
%  	 | 115 |  6729 |     0 | 1.252e-13 |
%  	 | 116 |  6846 |     0 | 2.220e-13 |
%  	 | 117 |  6963 |     0 | 1.394e-13 |
%  	 | 118 |  7082 |     0 | 2.376e-13 |
%  	 | 119 |  7201 |     0 | 3.775e-15 |
%  	 | 120 |  7322 |     0 | 2.167e-13 |
%  	 | 121 |  7443 |     0 | 7.550e-14 |
%  	 | 122 |  7566 |     0 | 2.438e-13 |
%  	 | 123 |  7689 |     0 | 1.372e-13 |
%  	 | 124 |  7814 |     0 | 3.366e-13 |
%  	 | 125 |  7939 |     0 | 1.146e-13 |
%  	 | 126 |  8066 |     0 | 2.327e-13 |
%  	 | 127 |  8193 |     0 | 3.118e-13 |
%  	 | 128 |  8322 |     0 | 2.260e-13 |
%  	 | 129 |  8451 |     0 | 2.589e-13 |
%  	 | 130 |  8582 |     0 | 1.092e-13 |
%  	 | 131 |  8713 |     0 | 3.118e-13 |
%  	 | 132 |  8846 |     0 | 2.633e-13 |
%  	 | 133 |  8979 |     0 | 2.709e-13 |
%  	 | 134 |  9114 |     0 | 1.652e-13 |
%  	 | 135 |  9249 |     0 | 1.785e-13 |
%  	 | 136 |  9386 |     0 | 3.095e-13 |
%  	 | 137 |  9523 |     0 | 2.434e-13 |
%  	 | 138 |  9662 |     0 | 1.394e-13 |
%  	 | 139 |  9801 |     0 | 2.540e-13 |
%  	 | 140 |  9942 |     0 | 2.238e-13 |
%  	 | 141 | 10083 |     0 | 3.628e-13 |
%  	 | 142 | 10226 |     0 | 9.237e-14 |
%  	 | 143 | 10369 |     0 | 5.684e-14 |
%  	 | 144 | 10514 |     0 | 1.972e-13 |
%  	 | 145 | 10659 |     0 | 4.352e-13 |
%  	 | 146 | 10806 |     0 | 2.873e-13 |
%  	 | 147 | 10953 |     0 | 1.554e-13 |
%  	 | 148 | 11102 |     0 | 2.247e-13 |
%  	 | 149 | 11251 |     0 | 4.423e-13 |
%  	 | 150 | 11402 |     0 | 2.851e-13 |
%    | 151 | 11553 |     0 |      |
%  	 | 152 | 11706 |     0 |      |
%  	 | 153 | 11859 |     0 |      |
%  	 | 154 | 12014 |     0 |      |
%  	 | 155 | 12169 |     0 |      |
%  	 | 156 | 12326 |     0 |      |
%  	 | 157 | 12483 |     0 |      |
%  	 | 158 | 12642 |     0 |      |
%  	 | 159 | 12801 |     0 |      |
%  	 | 160 | 12962 |     0 |      |
%  	 | 161 | 13123 |     0 |      |
%  	 | 162 | 13286 |     0 |      |
%  	 | 163 | 13449 |     0 |      |
%  	 | 164 | 13614 |     0 |      |
%  	 | 165 | 13779 |     0 |      |
%  	 | 166 | 13946 |     0 |      |
%  	 | 167 | 14113 |     0 |      |
%  	 | 168 | 14282 |     0 |      |
%  	 | 169 | 14451 |     0 |      |
%  	 | 170 | 14622 |     0 |      |
%  	 | 171 | 14793 |     0 |      |
%  	 | 172 | 14966 |     0 |      |
%  	 | 173 | 15139 |     0 |      |
%  	 | 174 | 15314 |     0 |      |
%  	 | 175 | 15489 |     0 |      |
%  	 | 176 | 15666 |     0 |      |
%  	 | 177 | 15843 |     0 |      |
%  	 | 178 | 16022 |     0 |      |
%  	 | 179 | 16201 |     0 |      |
%  	 | 180 | 16382 |     0 |      |
%--------------------------------------------------------------------------

% Troubleshooting.
if nargin < 1, deg=16; end

if deg > 180
    fprintf(2,'\n \t The ADE is too high. Setting deg=180.');
    deg=180;
end

degstr=num2str(deg);
if deg < 10
    degstr=strcat('00',degstr);
else
    if deg < 100,  degstr=strcat('0',degstr); end
end

% Getting file with rule data.
filename=strcat('sf',degstr,'.dat'); cd('RWS');
files=dir(filename);
fName=files.name; fFolder=files.folder;
fullname=fullfile(fFolder,fName);

% Loading data and assemblying results.
xyzwVALS=load(fullname);
xyz=xyzwVALS;
w=(4*pi/size(xyz,1))*ones(size(xyz,1),1);
xyzw=[xyz w];

% Back to main folder.
cd ..