
function [ETA,G,E,A,B]=compute_MZ_constant_lowmemory(deg,rule,dbox,...
    basis_indices,G)

%--------------------------------------------------------------------------
% OBJECT:
%--------------------------------------------------------------------------
% Low memory computation of MZ parameters on hypercubes. If the rule has
% large cardinality, this routine is faster than the classical one.
%--------------------------------------------------------------------------
% INPUT:
%--------------------------------------------------------------------------
% deg: MZ polynomial degree;
% rule: quadrature rule of which one needs to compute the MZ constants;
% dbox: bounding box of the domain, e.g. [-1 -1 -1; 1 1 1] for [-1,1]^3;
% basis_indices: sequence of indices of the orthnrm. polynomial basis;
% G: Gram matrix (if available).
%--------------------------------------------------------------------------
% OUTPUT:
%--------------------------------------------------------------------------
% ETA: MZ constant (relatively to integration error)
% A, B: MZ constants
% basis_indices: sequence of indices of the orthnrm. polynomial basis
% G: Gram matrix
%--------------------------------------------------------------------------
% Reference paper:
%--------------------------------------------------------------------------
% "On the role of weak Marcinkiewicz-Zygmund constants in polynomial 
% approximation by orthogonal bases"
% C. An, A. Sommariva and M. Vianello
%--------------------------------------------------------------------------
% License:
%--------------------------------------------------------------------------
% Copyright (C) 2025 Alvise Sommariva, Marco Vianello.
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation; either version 3 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%
% Authors:
%
% Alvise Sommariva <alvise@math.unipd.it>
% Marco Vianello <marcov@math.unipd.it>
% Date: December 27, 2025
%--------------------------------------------------------------------------
if nargin < 3, dbox=[-1 -1 -1; 1 1 1]; end

if nargin < 4, basis_indices=[]; end

if isempty(basis_indices)
    d=size(dbox,2);
    N = nchoosek(deg+d,d); basis_indices = zeros(N,d);
    for i=2:N
        basis_indices(i,:) = mono_next_grlex(d,basis_indices(i-1,:));
    end
    dimpoly=N;
else
    d=size(dbox,2);
    dimpoly=nchoosek(deg+d,d);
end



if nargin < 5, G=zeros(dimpoly,dimpoly); end

Vx=vandermonde_nlegendre_orthn(deg,rule(:,1),dbox(:,1));
Vy=vandermonde_nlegendre_orthn(deg,rule(:,2),dbox(:,2));
Vz=vandermonde_nlegendre_orthn(deg,rule(:,3),dbox(:,3));


for i=1:dimpoly
    vi=basis_indices(i,:)+1;
    term1=Vx(:,vi(1)).*Vy(:,vi(2)).*Vz(:,vi(3));
    for j=1:i
        vj=basis_indices(j,:)+1;
        term2=Vx(:,vj(1)).*Vy(:,vj(2)).*Vz(:,vj(3));
        w=rule(:,end); % equal weight rule
        G(i,j)=sum(w.*term1.*term2);
    end
end

GL=tril(G,-1);
G=G+GL';

E=eye(size(G))-G;
ETA=norm(E);

if nargout > 3
    if sum(isnan(G)) == 0
        lambda=eig(G);
        A=min(lambda);
        B=max(lambda);
    else
        A=NaN;
        B=NaN;
    end
end

A=abs(A); % there are negative A with A is close to machine precision.





