
function rule=cub_cube_mpx(ade,fam)

%--------------------------------------------------------------------------
% OBJECT.
%--------------------------------------------------------------------------
% Non-tensor Clenshaw-Curtis like formula for the Chebyshev measure
%
%           w(x,y,z)=1/(pi^3* (sqrt(1-x^2)*sqrt(1-y^2)*sqrt(1-z^2)) )
%
% in the unit-cube [-1,1]^3.
%--------------------------------------------------------------------------
% INPUT
%--------------------------------------------------------------------------
% ade: requested alg. degree of precision of the rule for integration of
%               int(f(x,y,z)*w(x,y,z))    x,y,z in [-1,1]^3;
%
% fam: parameter that allows to choose one of the 4 sub-grids below
%     (1 by default).
%--------------------------------------------------------------------------
% OUTPUT
%--------------------------------------------------------------------------
% nodes: cubature nodes;
% weights: cubature weights.
%--------------------------------------------------------------------------
% NOTE
%--------------------------------------------------------------------------
% The formula has been checked in terms of
%
% 1. moments computation w.r.t. tensorial Chebyshev orthogonal basis,
%
% 2. cardinality when "ade" is 2,6,10,... In this case, since it chooses a
% parameter "n" in the main theorem of [3] that is an even integer and its 
% asymptotic cardinality is a priori known, i.e. ((n+1)^3+/-(n+1))/4 and it
% is close to the cardinality of the implemented rule.
%
% Example: 
%
% ade th.  card
% 2     9    8
% 6    35   32
% 10   91   88
% 14  189  184
% 18  341  336
% 22  559  552
% 26  855  848
% 30  1241 1232
%
% In the first column of the table above we present the required ade, in 
% the second column the expected number of points, in the last column the 
% actual cardinality of the rule.
%--------------------------------------------------------------------------
% EXAMPLE
%--------------------------------------------------------------------------
% >> rule=cub_cube_mpx(3)
% rule =
%     1.0000    1.0000    1.0000    0.0625
%     1.0000   -1.0000    1.0000    0.0625
%    -1.0000    1.0000    1.0000    0.0625
%    -1.0000   -1.0000    1.0000    0.0625
%     1.0000    1.0000   -1.0000    0.0625
%     1.0000   -1.0000   -1.0000    0.0625
%    -1.0000    1.0000   -1.0000    0.0625
%    -1.0000   -1.0000   -1.0000    0.0625
%     0.0000    0.0000    0.0000    0.5000
% >>
%--------------------------------------------------------------------------
% AUTHORS
%--------------------------------------------------------------------------
% Stefano De Marchi, Marco Vianello
% University of Padova
% Revised by Alvise Sommariva
%--------------------------------------------------------------------------
% PAPER
%--------------------------------------------------------------------------
% [1] "Effective numerical integration on complex shaped elements by
%      discrete signed measures", 2025,
%      by L. Rinaldi, A. Sommariva, M. Vianello.
% [2] "ORTHOCUB: integral and  differential cubature rules by orthogonal
%      moments", 2025,
%      by L. Rinaldi, A. Sommariva, M. Vianello.
% [3]  New cubature formulae and hyperinterpolation in three variables
%      by Stefano De Marchi, Marco Vianello & Yuan Xu
%      BIT Numerical Mathematics (2006), DOI:10.1007/s10543-000-0000-x
%--------------------------------------------------------------------------
% First version: May 2008
% Revision: November 24, 2025
% Modified by Alvise Sommariva.
%--------------------------------------------------------------------------

if nargin < 1, ade=5; end
if nargin < 2, fam=1; end

% .......................... Nodes generation .............................


n=floor(ade/2)+1;

k=0:n; c=cos(k*pi/(n));   % Chebyshev-Lobatto points

% Here we extract the sub-vector E=EVEN and O=ODD of the Chebsyhev points
E=c(1:2:length(c)); O=c(2:2:length(c));

%--------------------------------------------------------------------------
% Generation of the nodes sub-grids.
%
% We have 4 possible families:
% 1) EEE OOO
% 2) EOO OEE
% 3) OEE EOO
% 4) EOE OEO
%--------------------------------------------------------------------------

switch(fam)
    case 1 % Default.
        [X1,Y1,Z1]=meshgrid(E,E,E);
        [X2,Y2,Z2]=meshgrid(O,O,O);
    case 2
        [X1,Y1,Z1]=meshgrid(E,E,O);
        [X2,Y2,Z2]=meshgrid(O,O,E);

    case 3
        [X1,Y1,Z1]=meshgrid(E,O,O);
        [X2,Y2,Z2]=meshgrid(O,E,E);
    case 4
        [X1,Y1,Z1]=meshgrid(O,O,O);
        [X2,Y2,Z2]=meshgrid(E,E,E);
end

EEE=[X1(:),Y1(:),Z1(:)]; OOO=[X2(:),Y2(:),Z2(:)]; XYZ=[EEE; OOO];

% .......................... Weights generation ...........................

X=XYZ(:,1); Y=XYZ(:,2); Z=XYZ(:,3);

Xtest= (X == -1) | (X == 1);
Ytest= (Y == -1) | (Y == 1);
Ztest= (Z == -1) | (Z == 1);

Wfactor=2.^(-(Xtest+Ytest+Ztest));
W0=(4/n^3)*ones(size(X));
W=Wfactor.*W0;

% .......................... Nodes/weights storage ........................

rule=[XYZ W];
