
function XYZW=cub_cube_prodrule(deg,alphabeta_math)

%--------------------------------------------------------------------------
% OBJECT:
%--------------------------------------------------------------------------
% Tensorial Gauss rule on the cube [-1,1]^3.
% The tensorial Jacobi weight functions are described by "alphabeta_math"
% that is a matrix 3 x 2.
% In particular
%
% * alphabeta_math(1,:) is the Jacobi exponend in the x-direction,
% * alphabeta_math(2,:) is the Jacobi exponend in the y-direction,
% * alphabeta_math(3,:) is the Jacobi exponend in the z-direction,
%
% If "alphabeta_math" is not declared, the Legendre weight will be
% considered.
%--------------------------------------------------------------------------
% Related paper:
%--------------------------------------------------------------------------
% "Integration and differentiation by orthogonal moments"
% L. Rinaldi, A. Sommariva and M. Vianello
%--------------------------------------------------------------------------
% Tests.
%--------------------------------------------------------------------------
% Tested on Matlab R2024B, on a PC running Intel(R) N150 (800 MHz) with 16
% GB of RAM.
%--------------------------------------------------------------------------
% License:
%--------------------------------------------------------------------------
% Copyright (C) 2025 Laura Rinaldi, Alvise Sommariva, Marco Vianello.
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation; either version 3 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%
% Authors:
%
% Laura Rinaldi    <laura.rinaldi@unipd.it>
% Alvise Sommariva <alvise@math.unipd.it>
% Marco Vianello   <marcov@math.unipd.it>
%
% Date: November 04, 2025
%--------------------------------------------------------------------------

if nargin < 2, alphabeta_math=[0 0; 0 0; 0 0]; end % default value

card_gs=ceil((deg+1)/2);

xw=gauss_jacobi(card_gs,alphabeta_math(1,1),alphabeta_math(1,2));
yw=gauss_jacobi(card_gs,alphabeta_math(2,1),alphabeta_math(2,2));
zw=gauss_jacobi(card_gs,alphabeta_math(3,1),alphabeta_math(3,2));

[X,Y,Z]=ndgrid(xw(:,1),yw(:,1),zw(:,1));
[WX,WY,WZ]=ndgrid(xw(:,2),yw(:,2),zw(:,2));

XYZW=[X(:) Y(:) Z(:) (WX(:)).*(WY(:)).*(WZ(:))];






function xw=gauss_jacobi(N,a,b)

%--------------------------------------------------------------------------
% OBJECT:
%--------------------------------------------------------------------------
% Gauss-Jacobi rule on (-1,1) where "a", "b" are Jacobi weight function 
% exponents.
%--------------------------------------------------------------------------
% Tests.
%--------------------------------------------------------------------------
% Tested on Matlab R2024B, on a PC running Intel(R) N150 (800 MHz) with 16
% GB of RAM.
%--------------------------------------------------------------------------
% License:
%--------------------------------------------------------------------------
% Copyright (C) 2025 Laura Rinaldi, Alvise Sommariva, Marco Vianello.
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation; either version 3 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%
% Authors:
%
% Laura Rinaldi    <laura.rinaldi@unipd.it>
% Alvise Sommariva <alvise@math.unipd.it>
% Marco Vianello   <marcov@math.unipd.it>
%
% Date: November 04, 2025
%--------------------------------------------------------------------------

ab=r_jacobi(N,a,b);
xw=gauss(N,ab);





function ab=r_jacobi(N,a,b)

% R_JACOBI Recurrence coefficients for monic Jacobi polynomials.
%
%    ab=R_JACOBI(n,a,b) generates the first n recurrence
%    coefficients for monic Jacobi polynomials with parameters
%    a and b. These are orthogonal on [-1,1] relative to the
%    weight function w(t)=(1-t)^a(1+t)^b. The n alpha-coefficients
%    are stored in the first column, the n beta-coefficients in
%    the second column, of the nx2 array ab. The call ab=
%    R_JACOBI(n,a) is the same as ab=R_JACOBI(n,a,a) and
%    ab=R_JACOBI(n) the same as ab=R_JACOBI(n,0,0).
%
%    Supplied by Dirk Laurie, 6-22-1998; edited by Walter
%    Gautschi, 4-4-2002.

if nargin<2, a=0; end;  if nargin<3, b=a; end
if((N<=0)|(a<=-1)|(b<=-1)) error('parameter(s) out of range'), end
nu=(b-a)/(a+b+2);
mu=2^(a+b+1)*gamma(a+1)*gamma(b+1)/gamma(a+b+2);
if N==1, ab=[nu mu]; return, end
N=N-1; n=1:N; nab=2*n+a+b;
A=[nu (b^2-a^2)*ones(1,N)./(nab.*(nab+2))];
n=2:N; nab=nab(n);
B1=4*(a+1)*(b+1)/((a+b+2)^2*(a+b+3));
B=4*(n+a).*(n+b).*n.*(n+a+b)./((nab.^2).*(nab+1).*(nab-1));
ab=[A' [mu; B1; B']];


function xw=gauss(N,ab)

%    Given a weight function w encoded by the nx2 array ab of the
%    first n recurrence coefficients for the associated orthogonal
%    polynomials, the first column of ab containing the n alpha-
%    coefficients and the second column the n beta-coefficients,
%    the call xw=GAUSS(n,ab) generates the nodes and weights xw of
%    the n-point Gauss quadrature rule for the weight function w.
%    The nodes, in increasing order, are stored in the first
%    column, the n corresponding weights in the second column, of
%    the nx2 array xw.
%    Supplied by Dirk Laurie, edited by Walter Gautschi.

N0=size(ab,1); if N0<N, error('input array ab too short'), end
J=zeros(N);
for n=1:N, J(n,n)=ab(n,1); end
for n=2:N
    J(n,n-1)=sqrt(ab(n,2));
    J(n-1,n)=J(n,n-1);
end
[V,D]=eig(J);
[D,I]=sort(diag(D));
V=V(:,I);
xw=[D ab(1,2)*V(1,:)'.^2];





