
function rule=cub_cube_prodrule_clenshawcurtis(ade)

%--------------------------------------------------------------------------
% OBJECT:
%--------------------------------------------------------------------------
% Tensorial Clenshaw-Curtis rule  on the cube [-1,1]^3.
% rule: matrix of nodes and weights; rule(:,1:3) are the nodes, while
%       rule(:,4) are the respective weights.
%--------------------------------------------------------------------------
% Related paper:
%--------------------------------------------------------------------------
%
%--------------------------------------------------------------------------
% Tests.
%--------------------------------------------------------------------------
% Tested on Matlab R2024B, on a PC running Intel(R) N150 (800 MHz) with 16
% GB of RAM.
%--------------------------------------------------------------------------
% License:
%--------------------------------------------------------------------------
% Copyright (C) 2025 Laura Rinaldi, Alvise Sommariva, Marco Vianello.
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation; either version 3 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%
% Authors:
%
% Laura Rinaldi    <laura.rinaldi@unipd.it>
% Alvise Sommariva <alvise@math.unipd.it>
% Marco Vianello   <marcov@math.unipd.it>
%
% Date: November 14, 2025
%--------------------------------------------------------------------------

xw=cub_interval_clenshawcurtis(ade); yw=xw; zw=xw;

[X,Y,Z]=ndgrid(xw(:,1),yw(:,1),zw(:,1));
[WX,WY,WZ]=ndgrid(xw(:,2),yw(:,2),zw(:,2));

rule=[X(:) Y(:) Z(:) (WX(:)).*(WY(:)).*(WZ(:))];





function rule=cub_interval_clenshawcurtis(ade)

%--------------------------------------------------------------------------
% Object
%--------------------------------------------------------------------------
% Clenshaw-Curtis quadrature rule on (-1,1) with ADE equal to "ade".
% The rule is w.r.t. Legendre weight function.
%--------------------------------------------------------------------------
% Input:
%--------------------------------------------------------------------------
% ade: algebraic degree of precision of the rule, i.e. number of points *1. 
%      Tested with n <= 14 million nodes.
% moms: weight function modified moments (w.r.t. Cheb. poly. type I).
%--------------------------------------------------------------------------
% Output:
%--------------------------------------------------------------------------
% rule: (n+1) x 2 matrix of nodes and weights.
%--------------------------------------------------------------------------
% Reference:
%--------------------------------------------------------------------------
% "Fast Construction of Fejer and Clenshaw-Curtis rules for general weight
% functions".
% A. Sommariva
%--------------------------------------------------------------------------
% License:
%--------------------------------------------------------------------------
% Copyright (C) 2025 Alvise Sommariva.
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation; either version 3 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%
% Authors:
%
% Alvise Sommariva <alvise@math.unipd.it>
%
% Date: November 04, 2025
%--------------------------------------------------------------------------

n=ade+1;

if nargin < 2
    moms=moms_cheb1type_legendre(n-1);
end

momsIIcc=compute_moments_IIwcc(n-3,moms);
theta=(1:n-2)'*pi/(n-1); xx=cos(theta);
w=((2*sin(theta)/(n-1)).*dst(momsIIcc))./(1-xx.^2);
w1=(2*sum(moms)-moms(1)-moms(end))/(2*(n-1));
wn=moms(1)-w1-sum(w);
x=[1;xx;-1]; w=[w1;w;wn];

rule=[x w];





%--------------------------------------------------------------------------
% Subroutines
%--------------------------------------------------------------------------

function [momsII,momsI]=compute_moments_IIwcc(n,momsI)

% COMPUTATION OF MOMENTS W.R.T. CHEBYSHEV SECOND TYPE FROM CHEBYSHEV FIRST
% TYPE.

momsI=momsI(1:n+3);
momsII=0.5*(momsI(1:end-2)-momsI(3:end));



function moms=moms_cheb1type_legendre(n)

moms=zeros(n+1,0);
m=0:2:n; m=m';
moms_even_degree=2./(1-m.^2);
moms(m+1,1)=moms_even_degree;

