

function [XYW,dbox]=cub_disk_lowcard(deg)

%--------------------------------------------------------------------------
% OBJECT.
% Low cardinality cubature rules on the unit-disk.
%--------------------------------------------------------------------------
% INPUT: 
% deg: degree of precision of the formula
%--------------------------------------------------------------------------
% OUTPUT:
% XYW: the nodes are (X,Y), with X=XYW(:,1), Y=XYW(:,2)), while the weights
%      are W=XYW(:,3).
% dbox: bounding box of the domain.
%--------------------------------------------------------------------------
% DATE:
% May 28, 2023
%--------------------------------------------------------------------------
% AUTHOR
%--------------------------------------------------------------------------
% Alvise Sommariva
%--------------------------------------------------------------------------
% NOTE
%--------------------------------------------------------------------------
% All these formula are taken from papers available in the web and are not
% computed directly by the author.
%--------------------------------------------------------------------------

% At higher degrees only few formulas are available.
% In this case we look for that with closer and higher degree
% of exactness.
% For degree of exactness higher that 77, we use a suitable low cardinality
% tensorial formula.

degH=[21 25 27 29 33 37 39 41 45 49 53 65 77];

if deg <= 19 | ismember(deg,degH)
    XYW=cub_unit_disk_sets_sub(deg);
else
    if deg <= 77
        % We look in this case for the closer ones between those with higher
        % degree of exactness.
        diff_deg=degH-deg;
        k=min(find(diff_deg>=0));
        deg=degH(k);
        XYW=cub_unit_disk_sets_sub(deg);
    else
        % For degree of exactness higher of 77, we use a suitable low 
        % cardinality tensorial formula.

        XYW = cub_circsegm(deg,pi,1);
    end
end

dbox=[-1 1; -1 1];



function XYW=cub_unit_disk_sets_sub(deg)

% .........................................................................
% Object:
% Low cardinality cubature sets for cubature on the unit-disk, with respect
% to the Legendre measure "w(x,y)=1".
%
% For a more complete list in Python see:
%          https://awesomeopensource.com/project/nschloe/quadpy
%
% Note: some sets were converted in Matlab from quadpy.
% .........................................................................

% .........................................................................
% | Deg| Card | momserr | max-norm |
% .........................................................................
% |  1 |    1 | 0.0e+00 | 0.00e+00 |
% |  2 |    3 | 2.8e-15 | 7.07e-01 |
% |  3 |    4 | 2.7e-15 | 7.07e-01 |
% |  4 |    6 | 6.7e-16 | 8.16e-01 |
% |  5 |    7 | 2.4e-15 | 8.16e-01 |
% |  6 |   12 | 6.2e-15 | 9.11e-01 |
% |  7 |   12 | 5.9e-15 | 9.11e-01 |
% |  8 |   16 | 3.2e-15 | 9.53e-01 |
% |  9 |   19 | 6.9e-14 | 9.19e-01 |
% | 10 |   28 | 1.4e-15 | 9.95e-01 |
% | 11 |   28 | 1.4e-15 | 9.95e-01 |
% | 12 |   36 | 1.8e-13 | 9.97e-01 |
% | 13 |   36 | 1.7e-15 | 9.97e-01 |
% | 14 |   44 | 2.0e-15 | 9.90e-01 |
% | 15 |   44 | 2.0e-15 | 9.90e-01 |
% | 16 |   57 | 1.1e-15 | 9.77e-01 |
% | 17 |   55 | 5.4e-15 | 9.83e-01 |
% | 18 |   72 | 1.6e-15 | 9.92e-01 |
% | 19 |   72 | 1.7e-15 | 9.92e-01 |
% | 25 |  117 | 3.2e-13 | 9.95e-01 |
% | 27 |  137 | 1.5e-13 | 9.99e-01 |
% | 29 |  155 | 2.9e-13 | 9.95e-01 |
% | 33 |  199 | 1.8e-13 | 9.96e-01 |
% | 37 |  247 | 1.7e-14 | 9.97e-01 |
% | 39 |  273 | 1.5e-13 | 9.99e-01 |
% | 41 |  295 | 1.8e-14 | 9.97e-01 |
% | 45 |  361 | 1.8e-13 | 9.98e-01 |
% | 49 |  425 | 1.9e-13 | 9.99e-01 |
% | 53 |  487 | 2.6e-14 | 9.99e-01 |
% | 65 |  733 | 2.9e-14 | 9.99e-01 |
% | 77 | 1021 | 3.4e-14 | 9.99e-01 |
%------------------------------------------
% .........................................................................
% Legend:
% Deg: degree of exactness
% Card: set cardinality
% momserr: maximum error in matching the moments of Logan-Shepp polynomial
%       basis
% max-norm: max norm of a point of the pointset ( max-norm<=1 implies
%          points inside the unit-disk)
% .........................................................................



switch deg

    case {21, 25, 27, 29, 33, 37, 39, 41, 45, 49, 53, 65, 77}

        %------------------------------------------
        %  deg   card    dim     err     larg.rad.
        %------------------------------------------
        % | 17 |   55 |  171 | 5.4e-15 | 9.83e-01 |
        % | 21 |   85 |  253 | 1.0e-14 | 9.90e-01 |
        % | 25 |  117 |  351 | 3.2e-13 | 9.95e-01 |
        % | 27 |  137 |  406 | 1.5e-13 | 9.99e-01 |
        % | 29 |  155 |  465 | 2.9e-13 | 9.95e-01 |
        % | 33 |  199 |  595 | 1.8e-13 | 9.96e-01 |
        % | 37 |  247 |  741 | 1.7e-14 | 9.97e-01 |
        % | 39 |  273 |  820 | 1.5e-13 | 9.99e-01 |
        % | 41 |  295 |  903 | 1.8e-14 | 9.97e-01 |
        % | 45 |  361 | 1081 | 1.8e-13 | 9.98e-01 |
        % | 49 |  425 | 1275 | 1.9e-13 | 9.99e-01 |
        % | 53 |  487 | 1485 | 2.6e-14 | 9.99e-01 |
        % | 65 |  733 | 2211 | 2.9e-14 | 9.99e-01 |
        % | 77 | 1021 | 3081 | 3.4e-14 | 9.99e-01 |
        %------------------------------------------

        XYW=unit_disk_rules_takaki_forbes_rolland(deg);

    case {1,13,18,19}

        % ...................................
        % | Deg | Card | type |
        % ...................................
        % |  1 |    1 | 0.0e+00 | 0.00e+00 |
        % | 13 |   36 | 1.7e-15 | 9.97e-01 |
        % | 18 |   72 | 1.6e-15 | 9.92e-01 |
        % | 19 |   72 | 1.7e-15 | 9.92e-01 |
        % ...................................

        XYW=cub_rules_nschloe(deg);


    case {9}

        % ..........................
        % | Deg | Card | type |
        % .........................
        % |  3  |   4  | PI
        % |  5  |   7  | PI
        % |  7  |  14  | PI
        % |  9  |  19  | PI
        % | 11  |  30  | PI
        % | 13  |  37  | PI
        % | 15  |  52  | PI *
        % | 17  |  61  | PI *
        % .........................

        XYW=cub_unit_disk_luo_meng(deg);

    case {16,17}

        % ..........................
        % | Deg | Card | type |
        % .........................
        % | 17  |  57  |  PI  |
        % | 19  |  69  |  PO  |
        % | 21  |  88  |  PO  |
        % .........................

        if deg == 16, deg=17; end

        XYW=cub_unit_disk_cools_kim(deg);

    case {8}

        % ..........................
        % | Deg | Card | type |
        % ..........................
        % | 6   |  10  |  PO  |
        % | 6   |  11  |  PO  |
        % | 8   |  16  |  PI  |
        % ..........................

        XYW=cub_unit_disk_wissmann_becker(deg);

    case {10,11,14,15}
        % .........................
        % | Deg | Card | type |
        % .........................
        % |  9  |  20  |  PO  |
        % |  9  |  21  |  PO  |
        % | 11  |  28  |  PI  |
        % | 11  |  28  |  PI  |
        % | 13  |  37  |  PI  |
        % | 15  |  44  |  PI  |
        % .........................
        if deg == 10, deg=11; end
        if deg == 14, deg=15; end
        XYW=cub_unit_disk_rabinowitz_richter(deg);

    case {6,12}
        % .........................
        % | Deg | Card | type|
        % .........................
        % |  1  |  1  |  PI  |
        % |  2  |  4  |  PI  |
        % |  3  |  4  |  PI  |
        % |  4  |  8  |  PO  |
        % |  5  |  8  |  PO  |
        % |  6  | 12  |  PI  |
        % |  7  | 12  |  PI  |
        % |  8  | 20  |  PO  |
        % |  9  | 20  |  PO  |
        % | 12  | 36  |  PI  |
        % | 13  | 36  |  PI  |
        % | 14  | 44  |  PI  | *
        % | 15  | 44  |  PI  | *
        % .........................
        XYW=cub_rules_disk_2012(deg);

        %     case {11}
        %
        %         % .........................
        %         % | Deg | Card | type|
        %         % .....................
        %         % |  11 |  26 |  PI* |
        %         % .........................
        %
        %         XYW=cub_rules_piessens_haegemans(deg);
        %

    case {2,3,4,5,7}

        % .........................
        % | Deg | Card | type|
        % .........................
        % |  2  |  3  |  PI  |
        % |  3  |  4  |  PI  |
        % |  4  |  6  |  PI  |
        % |  5  |  7  |  PI  |
        % |  7  | 12  |  PI  |
        % .........................


        XYW=cub_rules_stroud_1971(deg);

    otherwise

        warning('Set not implemented.');
        
        XYW=cub_rules_nschloe(19);


end

















function XYW=cub_unit_disk_luo_meng(deg)

%--------------------------------------------------------------------------
% Reference paper:
% Zhongxuan Luo, Zhaoliang Meng,
% "Cubature formulas over the n-sphere"
% Journal of Computational and Applied Mathematics 202 (2007) 511 ? 522
%--------------------------------------------------------------------------
% | Deg | Card |
% ..................
% |  3  |   4  | PI
% |  5  |   7  | PI
% |  7  |  14  | PI
% |  9  |  19  | PI
% | 11  |  30  | PI
% | 13  |  37  | PI
% | 15  |  52  | PI *
% | 17  |  61  | PI *
%
% Note:
% * bad moment matching.
%--------------------------------------------------------------------------

switch deg

    case 3
        % Generators of symmetric 4-point cubature of degree 3 for the unit disk
        XW0=[0.5 0.5 0.78539816339745];

        % deg:  3
        % moment matching: 4.0e-15
        % cardinality:   4
        % max norm point: 7.07e-01

        XYW=[ -0.500000000000000  -0.500000000000000   0.785398163397450
            -0.500000000000000   0.500000000000000   0.785398163397450
            0.500000000000000  -0.500000000000000   0.785398163397450
            0.500000000000000   0.500000000000000   0.785398163397450];

    case 5
        % Generators of symmetric 7-point cubature of degree 5 for the unit disk
        XW0=[                   0   0.000000000000000 7.853981633974501e-01
            8.164965809277300e-01   0.000000000000000 3.926990816987200e-01
            4.082482904638600e-01     7.071067811865500e-01     3.926990816987200e-01];

        % deg:  5
        % moment matching: 4.7e-14
        % cardinality:   7
        % max norm point: 8.16e-01

        XYW=[
            -8.1649658092773003e-01 0.0000000000000000e+00 3.9269908169871998e-01
            -4.0824829046386002e-01 -7.0710678118655002e-01 3.9269908169871998e-01
            -4.0824829046386002e-01 7.0710678118655002e-01 3.9269908169871998e-01
            0.0000000000000000e+00 0.0000000000000000e+00 7.8539816339745006e-01
            4.0824829046386002e-01 -7.0710678118655002e-01 3.9269908169871998e-01
            4.0824829046386002e-01 7.0710678118655002e-01 3.9269908169871998e-01
            8.1649658092773003e-01 0.0000000000000000e+00 3.9269908169871998e-01];


    case 7
        % Generators of symmetric 14-point cubature of degree 7 for the unit disk
        XW0=[2.298504216905000e-01     3.981126085090600e-01     2.617993877991500e-01
            4.597008433809800e-01   0.000000000000000 2.617993877991500e-01
            3.370397518064200e-01     8.216321198061100e-01     1.950140767779300e-01
            8.193205902590500e-01     3.426206429454800e-01     1.976850049207900e-01];

        % deg:  7
        % moment matching: 3.8e-14
        % cardinality:  14
        % max norm point: 8.88e-01
        XYW=[
            -8.1932059025905002e-01 -3.4262064294548000e-01 1.9768500492079000e-01
            -8.1932059025905002e-01 3.4262064294548000e-01 1.9768500492079000e-01
            -4.5970084338098000e-01 0.0000000000000000e+00 2.6179938779915002e-01
            -3.3703975180641998e-01 -8.2163211980610995e-01 1.9501407677793001e-01
            -3.3703975180641998e-01 8.2163211980610995e-01 1.9501407677793001e-01
            -2.2985042169049999e-01 -3.9811260850906000e-01 2.6179938779915002e-01
            -2.2985042169049999e-01 3.9811260850906000e-01 2.6179938779915002e-01
            2.2985042169049999e-01 -3.9811260850906000e-01 2.6179938779915002e-01
            2.2985042169049999e-01 3.9811260850906000e-01 2.6179938779915002e-01
            3.3703975180641998e-01 -8.2163211980610995e-01 1.9501407677793001e-01
            3.3703975180641998e-01 8.2163211980610995e-01 1.9501407677793001e-01
            4.5970084338098000e-01 0.0000000000000000e+00 2.6179938779915002e-01
            8.1932059025905002e-01 -3.4262064294548000e-01 1.9768500492079000e-01
            8.1932059025905002e-01 3.4262064294548000e-01 1.9768500492079000e-01];

    case 9
        % Generators of symmetric 19-point cubature of degree 9 for the unit disk
        XW0=[
            0.000000000000000   0.000000000000000 3.490658503988700e-01
            2.280263556769600e-01     5.505043204538600e-01     2.012527133278100e-01
            5.505043204538600e-01     2.280263556769700e-01     2.012527133278100e-01
            2.795938737705800e-01     8.756576043341801e-01     1.166103997667300e-01
            7.393051718617600e-01     5.462388096215500e-01     1.189116253533400e-01
            9.192110607898000e-01     0.000000000000000 1.202084980440900e-01];

        % deg:  9
        % moment matching: 6.9e-14
        % cardinality:  19
        % max norm point: 9.19e-01
        XYW=[
            -9.1921106078979997e-01 0.0000000000000000e+00 1.2020849804409001e-01
            -7.3930517186176004e-01 -5.4623880962155003e-01 1.1891162535334000e-01
            -7.3930517186176004e-01 5.4623880962155003e-01 1.1891162535334000e-01
            -5.5050432045386000e-01 -2.2802635567696999e-01 2.0125271332781000e-01
            -5.5050432045386000e-01 2.2802635567696999e-01 2.0125271332781000e-01
            -2.7959387377057998e-01 -8.7565760433418005e-01 1.1661039976673000e-01
            -2.7959387377057998e-01 8.7565760433418005e-01 1.1661039976673000e-01
            -2.2802635567696000e-01 -5.5050432045386000e-01 2.0125271332781000e-01
            -2.2802635567696000e-01 5.5050432045386000e-01 2.0125271332781000e-01
            0.0000000000000000e+00 0.0000000000000000e+00 3.4906585039887000e-01
            2.2802635567696000e-01 -5.5050432045386000e-01 2.0125271332781000e-01
            2.2802635567696000e-01 5.5050432045386000e-01 2.0125271332781000e-01
            2.7959387377057998e-01 -8.7565760433418005e-01 1.1661039976673000e-01
            2.7959387377057998e-01 8.7565760433418005e-01 1.1661039976673000e-01
            5.5050432045386000e-01 -2.2802635567696999e-01 2.0125271332781000e-01
            5.5050432045386000e-01 2.2802635567696999e-01 2.0125271332781000e-01
            7.3930517186176004e-01 -5.4623880962155003e-01 1.1891162535334000e-01
            7.3930517186176004e-01 5.4623880962155003e-01 1.1891162535334000e-01
            9.1921106078979997e-01 0.0000000000000000e+00 1.2020849804409001e-01];

    case 11
        % Generators of symmetric 30-point cubature of degree 11 for the unit disk
        XW0=[1.284709179904800e-01     3.101562325827800e-01     1.090830782496500e-01
            3.101562325828800e-01     1.284709179902300e-01     1.090830782496500e-01
            2.182525075456800e-01     6.725814768115600e-01     1.394804395076700e-01
            5.718063589988400e-01     4.159777491747400e-01     1.396823099601400e-01
            7.071067811865500e-01   0.000000000000000 1.398062018621000e-01
            2.365511901974600e-01     9.117795068090200e-01     7.075156570176000e-02
            6.559028277674500e-01     6.760841775603100e-01     7.284300646890000e-02
            9.080809097354000e-01     2.503745114720700e-01     7.457158432862999e-02];

        % deg: 11
        % moment matching: 3.0e-13
        % cardinality:  30
        % max norm point: 9.42e-01
        XYW=[
            -9.0808090973540001e-01 -2.5037451147207002e-01 7.4571584328629995e-02
            -9.0808090973540001e-01 2.5037451147207002e-01 7.4571584328629995e-02
            -7.0710678118655002e-01 0.0000000000000000e+00 1.3980620186209999e-01
            -6.5590282776745001e-01 -6.7608417756031003e-01 7.2843006468900004e-02
            -6.5590282776745001e-01 6.7608417756031003e-01 7.2843006468900004e-02
            -5.7180635899884003e-01 -4.1597774917474001e-01 1.3968230996014000e-01
            -5.7180635899884003e-01 4.1597774917474001e-01 1.3968230996014000e-01
            -3.1015623258288000e-01 -1.2847091799023000e-01 1.0908307824965000e-01
            -3.1015623258288000e-01 1.2847091799023000e-01 1.0908307824965000e-01
            -2.3655119019745999e-01 -9.1177950680901998e-01 7.0751565701760000e-02
            -2.3655119019745999e-01 9.1177950680901998e-01 7.0751565701760000e-02
            -2.1825250754568001e-01 -6.7258147681156000e-01 1.3948043950767000e-01
            -2.1825250754568001e-01 6.7258147681156000e-01 1.3948043950767000e-01
            -1.2847091799047999e-01 -3.1015623258278002e-01 1.0908307824965000e-01
            -1.2847091799047999e-01 3.1015623258278002e-01 1.0908307824965000e-01
            1.2847091799047999e-01 -3.1015623258278002e-01 1.0908307824965000e-01
            1.2847091799047999e-01 3.1015623258278002e-01 1.0908307824965000e-01
            2.1825250754568001e-01 -6.7258147681156000e-01 1.3948043950767000e-01
            2.1825250754568001e-01 6.7258147681156000e-01 1.3948043950767000e-01
            2.3655119019745999e-01 -9.1177950680901998e-01 7.0751565701760000e-02
            2.3655119019745999e-01 9.1177950680901998e-01 7.0751565701760000e-02
            3.1015623258288000e-01 -1.2847091799023000e-01 1.0908307824965000e-01
            3.1015623258288000e-01 1.2847091799023000e-01 1.0908307824965000e-01
            5.7180635899884003e-01 -4.1597774917474001e-01 1.3968230996014000e-01
            5.7180635899884003e-01 4.1597774917474001e-01 1.3968230996014000e-01
            6.5590282776745001e-01 -6.7608417756031003e-01 7.2843006468900004e-02
            6.5590282776745001e-01 6.7608417756031003e-01 7.2843006468900004e-02
            7.0710678118655002e-01 0.0000000000000000e+00 1.3980620186209999e-01
            9.0808090973540001e-01 -2.5037451147207002e-01 7.4571584328629995e-02
            9.0808090973540001e-01 2.5037451147207002e-01 7.4571584328629995e-02];

    case 13
        % Generators of symmetric 37-point cubature of degree 13 for the unit disk
        XW0=[                   0   0.000000000000000 1.963495408493600e-01
            4.608042298407800e-01   0.000000000000000 1.033094899824100e-01
            1.423963381006800e-01     4.382508655264300e-01     1.033094899824100e-01
            3.727984530210700e-01     2.708539304943700e-01     1.033094899824100e-01
            1.982017738364800e-01     7.424615763844900e-01     1.013019642883700e-01
            5.423787774928500e-01     5.443880943634100e-01     1.016315917682400e-01
            7.420938368732500e-01     1.995742288824100e-01     1.019528814157100e-01
            2.043567172737200e-01     9.325504665123500e-01     4.772103677723000e-02
            5.813572776870600e-01     7.572554101276400e-01     4.919131994151000e-02
            8.542887598558900e-01     4.261487478230800e-01     5.064404906285000e-02
            9.546790248493400e-01   0.000000000000000 5.118841995037000e-02];

        % deg: 13
        % moment matching: 8.6e-14
        % cardinality:  37
        % max norm point: 9.55e-01
        XYW=[
            -9.5467902484933997e-01 0.0000000000000000e+00 5.1188419950369997e-02
            -8.5428875985589003e-01 -4.2614874782307999e-01 5.0644049062850000e-02
            -8.5428875985589003e-01 4.2614874782307999e-01 5.0644049062850000e-02
            -7.4209383687325003e-01 -1.9957422888241000e-01 1.0195288141571000e-01
            -7.4209383687325003e-01 1.9957422888241000e-01 1.0195288141571000e-01
            -5.8135727768706003e-01 -7.5725541012763997e-01 4.9191319941509999e-02
            -5.8135727768706003e-01 7.5725541012763997e-01 4.9191319941509999e-02
            -5.4237877749285002e-01 -5.4438809436340996e-01 1.0163159176824001e-01
            -5.4237877749285002e-01 5.4438809436340996e-01 1.0163159176824001e-01
            -4.6080422984078001e-01 0.0000000000000000e+00 1.0330948998241000e-01
            -3.7279845302107001e-01 -2.7085393049437001e-01 1.0330948998241000e-01
            -3.7279845302107001e-01 2.7085393049437001e-01 1.0330948998241000e-01
            -2.0435671727372001e-01 -9.3255046651234996e-01 4.7721036777229997e-02
            -2.0435671727372001e-01 9.3255046651234996e-01 4.7721036777229997e-02
            -1.9820177383647999e-01 -7.4246157638448995e-01 1.0130196428837000e-01
            -1.9820177383647999e-01 7.4246157638448995e-01 1.0130196428837000e-01
            -1.4239633810068000e-01 -4.3825086552643000e-01 1.0330948998241000e-01
            -1.4239633810068000e-01 4.3825086552643000e-01 1.0330948998241000e-01
            0.0000000000000000e+00 0.0000000000000000e+00 1.9634954084935999e-01
            1.4239633810068000e-01 -4.3825086552643000e-01 1.0330948998241000e-01
            1.4239633810068000e-01 4.3825086552643000e-01 1.0330948998241000e-01
            1.9820177383647999e-01 -7.4246157638448995e-01 1.0130196428837000e-01
            1.9820177383647999e-01 7.4246157638448995e-01 1.0130196428837000e-01
            2.0435671727372001e-01 -9.3255046651234996e-01 4.7721036777229997e-02
            2.0435671727372001e-01 9.3255046651234996e-01 4.7721036777229997e-02
            3.7279845302107001e-01 -2.7085393049437001e-01 1.0330948998241000e-01
            3.7279845302107001e-01 2.7085393049437001e-01 1.0330948998241000e-01
            4.6080422984078001e-01 0.0000000000000000e+00 1.0330948998241000e-01
            5.4237877749285002e-01 -5.4438809436340996e-01 1.0163159176824001e-01
            5.4237877749285002e-01 5.4438809436340996e-01 1.0163159176824001e-01
            5.8135727768706003e-01 -7.5725541012763997e-01 4.9191319941509999e-02
            5.8135727768706003e-01 7.5725541012763997e-01 4.9191319941509999e-02
            7.4209383687325003e-01 -1.9957422888241000e-01 1.0195288141571000e-01
            7.4209383687325003e-01 1.9957422888241000e-01 1.0195288141571000e-01
            8.5428875985589003e-01 -4.2614874782307999e-01 5.0644049062850000e-02
            8.5428875985589003e-01 4.2614874782307999e-01 5.0644049062850000e-02
            9.5467902484933997e-01 0.0000000000000000e+00 5.1188419950369997e-02];

    case 15
        % Generators of symmetric 52-point cubature of degree 15 for the unit disk
        XW0=[2.634992299855400e-01   0.000000000000000 5.464091129997000e-02
            8.142573996847000e-02     2.506026597495700e-01     5.464091129997000e-02
            2.131753549666300e-01     1.548809615085900e-01     5.464091129997000e-02
            1.486565937957200e-01     5.548970132633900e-01     8.535187281584000e-02
            4.061700442289200e-01     4.062454595175800e-01     8.536560703414001e-02
            5.548832113068500e-01     1.487081034018400e-01     8.537932359880999e-02
            1.804778332686200e-01     7.983847903680800e-01     7.254080017264000e-02
            5.073913002056500e-01     6.422963414717900e-01     7.303155482639000e-02
            7.361773352966400e-01     3.578176250381700e-01     7.360272502709000e-02
            8.185294874300000e-01   0.000000000000000 7.384344684536000e-02
            1.788529987522800e-01     9.479344706435900e-01     3.255139877430000e-02
            5.177175896494600e-01     8.139635453535899e-01     3.364947668306000e-02
            7.906275860164400e-01     5.526989922433700e-01     3.485759424523000e-02
            9.445200476315900e-01     1.960868058260100e-01     3.554380854735000e-02];

        % deg: 15
        % moment matching: 2.1e-10
        % cardinality:  52
        % max norm point: 9.65e-01
        XYW=[
            -9.4452004763159003e-01 -1.9608680582601001e-01 3.5543808547350002e-02
            -9.4452004763159003e-01 1.9608680582601001e-01 3.5543808547350002e-02
            -8.1852948742999998e-01 0.0000000000000000e+00 7.3843446845359997e-02
            -7.9062758601644001e-01 -5.5269899224337005e-01 3.4857594245230002e-02
            -7.9062758601644001e-01 5.5269899224337005e-01 3.4857594245230002e-02
            -7.3617733529663998e-01 -3.5781762503817000e-01 7.3602725027090005e-02
            -7.3617733529663998e-01 3.5781762503817000e-01 7.3602725027090005e-02
            -5.5488321130685003e-01 -1.4870810340184001e-01 8.5379323598809995e-02
            -5.5488321130685003e-01 1.4870810340184001e-01 8.5379323598809995e-02
            -5.1771758964946002e-01 -8.1396354535358995e-01 3.3649476683060002e-02
            -5.1771758964946002e-01 8.1396354535358995e-01 3.3649476683060002e-02
            -5.0739130020565004e-01 -6.4229634147179004e-01 7.3031554826389997e-02
            -5.0739130020565004e-01 6.4229634147179004e-01 7.3031554826389997e-02
            -4.0617004422892000e-01 -4.0624545951758001e-01 8.5365607034140006e-02
            -4.0617004422892000e-01 4.0624545951758001e-01 8.5365607034140006e-02
            -2.6349922998553998e-01 0.0000000000000000e+00 5.4640911299970001e-02
            -2.1317535496663001e-01 -1.5488096150858999e-01 5.4640911299970001e-02
            -2.1317535496663001e-01 1.5488096150858999e-01 5.4640911299970001e-02
            -1.8047783326861999e-01 -7.9838479036807997e-01 7.2540800172640002e-02
            -1.8047783326861999e-01 7.9838479036807997e-01 7.2540800172640002e-02
            -1.7885299875228000e-01 -9.4793447064359004e-01 3.2551398774300001e-02
            -1.7885299875228000e-01 9.4793447064359004e-01 3.2551398774300001e-02
            -1.4865659379571999e-01 -5.5489701326339003e-01 8.5351872815840002e-02
            -1.4865659379571999e-01 5.5489701326339003e-01 8.5351872815840002e-02
            -8.1425739968469996e-02 -2.5060265974957002e-01 5.4640911299970001e-02
            -8.1425739968469996e-02 2.5060265974957002e-01 5.4640911299970001e-02
            8.1425739968469996e-02 -2.5060265974957002e-01 5.4640911299970001e-02
            8.1425739968469996e-02 2.5060265974957002e-01 5.4640911299970001e-02
            1.4865659379571999e-01 -5.5489701326339003e-01 8.5351872815840002e-02
            1.4865659379571999e-01 5.5489701326339003e-01 8.5351872815840002e-02
            1.7885299875228000e-01 -9.4793447064359004e-01 3.2551398774300001e-02
            1.7885299875228000e-01 9.4793447064359004e-01 3.2551398774300001e-02
            1.8047783326861999e-01 -7.9838479036807997e-01 7.2540800172640002e-02
            1.8047783326861999e-01 7.9838479036807997e-01 7.2540800172640002e-02
            2.1317535496663001e-01 -1.5488096150858999e-01 5.4640911299970001e-02
            2.1317535496663001e-01 1.5488096150858999e-01 5.4640911299970001e-02
            2.6349922998553998e-01 0.0000000000000000e+00 5.4640911299970001e-02
            4.0617004422892000e-01 -4.0624545951758001e-01 8.5365607034140006e-02
            4.0617004422892000e-01 4.0624545951758001e-01 8.5365607034140006e-02
            5.0739130020565004e-01 -6.4229634147179004e-01 7.3031554826389997e-02
            5.0739130020565004e-01 6.4229634147179004e-01 7.3031554826389997e-02
            5.1771758964946002e-01 -8.1396354535358995e-01 3.3649476683060002e-02
            5.1771758964946002e-01 8.1396354535358995e-01 3.3649476683060002e-02
            5.5488321130685003e-01 -1.4870810340184001e-01 8.5379323598809995e-02
            5.5488321130685003e-01 1.4870810340184001e-01 8.5379323598809995e-02
            7.3617733529663998e-01 -3.5781762503817000e-01 7.3602725027090005e-02
            7.3617733529663998e-01 3.5781762503817000e-01 7.3602725027090005e-02
            7.9062758601644001e-01 -5.5269899224337005e-01 3.4857594245230002e-02
            7.9062758601644001e-01 5.5269899224337005e-01 3.4857594245230002e-02
            8.1852948742999998e-01 0.0000000000000000e+00 7.3843446845359997e-02
            9.4452004763159003e-01 -1.9608680582601001e-01 3.5543808547350002e-02
            9.4452004763159003e-01 1.9608680582601001e-01 3.5543808547350002e-02];

    case 17
        % Generators of symmetric 61-point cubature of degree 17 for the unit disk
        XW0=[                   0   0.000000000000000 1.256637061435900e-01
            9.675812994165001e-02     3.611062567084300e-01     5.840846471928000e-02
            2.643481269495700e-01     2.643481267609700e-01     5.840846471928000e-02
            3.611062567431600e-01     9.675812981204999e-02     5.840846471928000e-02
            1.434763933416600e-01     6.291455254428500e-01     6.992018223250999e-02
            4.021281217951800e-01     5.046806329675900e-01     6.996062112205000e-02
            5.813007507160900e-01     2.801767385918900e-01     7.001072267677000e-02
            6.452980455813300e-01   0.000000000000000 7.003290482486001e-02
            1.635987747573000e-01     8.345013045285100e-01     5.451221994169000e-02
            4.678232168068800e-01     7.101397215888900e-01     5.497050073244000e-02
            7.040577444714600e-01     4.769273307451000e-01     5.555796700074000e-02
            8.336158413527900e-01     1.680524186304800e-01     5.593580988433000e-02
            1.591209167698700e-01     9.579020501762100e-01     2.373604417773000e-02
            4.653016348962200e-01     8.522852764470300e-01     2.450554997213000e-02
            7.277990491393400e-01     6.428097292024700e-01     2.542637441947000e-02
            9.071498706133100e-01     3.463739541763000e-01     2.606318434828000e-02
            9.710282199223000e-01   0.000000000000000 2.628242756629000e-02];

        % deg: 17
        % moment matching: 1.4e-10
        % cardinality:  61
        % max norm point: 9.71e-01
        XYW=[
            -9.7102821992230004e-01 0.0000000000000000e+00 2.6282427566290002e-02
            -9.0714987061330998e-01 -3.4637395417630001e-01 2.6063184348280002e-02
            -9.0714987061330998e-01 3.4637395417630001e-01 2.6063184348280002e-02
            -8.3361584135278999e-01 -1.6805241863048001e-01 5.5935809884330001e-02
            -8.3361584135278999e-01 1.6805241863048001e-01 5.5935809884330001e-02
            -7.2779904913934002e-01 -6.4280972920247004e-01 2.5426374419469999e-02
            -7.2779904913934002e-01 6.4280972920247004e-01 2.5426374419469999e-02
            -7.0405774447145997e-01 -4.7692733074510002e-01 5.5557967000739998e-02
            -7.0405774447145997e-01 4.7692733074510002e-01 5.5557967000739998e-02
            -6.4529804558132997e-01 0.0000000000000000e+00 7.0032904824860007e-02
            -5.8130075071609000e-01 -2.8017673859189002e-01 7.0010722676769999e-02
            -5.8130075071609000e-01 2.8017673859189002e-01 7.0010722676769999e-02
            -4.6782321680687999e-01 -7.1013972158889005e-01 5.4970500732440003e-02
            -4.6782321680687999e-01 7.1013972158889005e-01 5.4970500732440003e-02
            -4.6530163489622001e-01 -8.5228527644703000e-01 2.4505549972129999e-02
            -4.6530163489622001e-01 8.5228527644703000e-01 2.4505549972129999e-02
            -4.0212812179517998e-01 -5.0468063296759003e-01 6.9960621122049996e-02
            -4.0212812179517998e-01 5.0468063296759003e-01 6.9960621122049996e-02
            -3.6110625674315999e-01 -9.6758129812049995e-02 5.8408464719279997e-02
            -3.6110625674315999e-01 9.6758129812049995e-02 5.8408464719279997e-02
            -2.6434812694956999e-01 -2.6434812676097003e-01 5.8408464719279997e-02
            -2.6434812694956999e-01 2.6434812676097003e-01 5.8408464719279997e-02
            -1.6359877475730000e-01 -8.3450130452851001e-01 5.4512219941689997e-02
            -1.6359877475730000e-01 8.3450130452851001e-01 5.4512219941689997e-02
            -1.5912091676987000e-01 -9.5790205017621000e-01 2.3736044177729999e-02
            -1.5912091676987000e-01 9.5790205017621000e-01 2.3736044177729999e-02
            -1.4347639334166001e-01 -6.2914552544284996e-01 6.9920182232509995e-02
            -1.4347639334166001e-01 6.2914552544284996e-01 6.9920182232509995e-02
            -9.6758129941650006e-02 -3.6110625670843000e-01 5.8408464719279997e-02
            -9.6758129941650006e-02 3.6110625670843000e-01 5.8408464719279997e-02
            0.0000000000000000e+00 0.0000000000000000e+00 1.2566370614358999e-01
            9.6758129941650006e-02 -3.6110625670843000e-01 5.8408464719279997e-02
            9.6758129941650006e-02 3.6110625670843000e-01 5.8408464719279997e-02
            1.4347639334166001e-01 -6.2914552544284996e-01 6.9920182232509995e-02
            1.4347639334166001e-01 6.2914552544284996e-01 6.9920182232509995e-02
            1.5912091676987000e-01 -9.5790205017621000e-01 2.3736044177729999e-02
            1.5912091676987000e-01 9.5790205017621000e-01 2.3736044177729999e-02
            1.6359877475730000e-01 -8.3450130452851001e-01 5.4512219941689997e-02
            1.6359877475730000e-01 8.3450130452851001e-01 5.4512219941689997e-02
            2.6434812694956999e-01 -2.6434812676097003e-01 5.8408464719279997e-02
            2.6434812694956999e-01 2.6434812676097003e-01 5.8408464719279997e-02
            3.6110625674315999e-01 -9.6758129812049995e-02 5.8408464719279997e-02
            3.6110625674315999e-01 9.6758129812049995e-02 5.8408464719279997e-02
            4.0212812179517998e-01 -5.0468063296759003e-01 6.9960621122049996e-02
            4.0212812179517998e-01 5.0468063296759003e-01 6.9960621122049996e-02
            4.6530163489622001e-01 -8.5228527644703000e-01 2.4505549972129999e-02
            4.6530163489622001e-01 8.5228527644703000e-01 2.4505549972129999e-02
            4.6782321680687999e-01 -7.1013972158889005e-01 5.4970500732440003e-02
            4.6782321680687999e-01 7.1013972158889005e-01 5.4970500732440003e-02
            5.8130075071609000e-01 -2.8017673859189002e-01 7.0010722676769999e-02
            5.8130075071609000e-01 2.8017673859189002e-01 7.0010722676769999e-02
            6.4529804558132997e-01 0.0000000000000000e+00 7.0032904824860007e-02
            7.0405774447145997e-01 -4.7692733074510002e-01 5.5557967000739998e-02
            7.0405774447145997e-01 4.7692733074510002e-01 5.5557967000739998e-02
            7.2779904913934002e-01 -6.4280972920247004e-01 2.5426374419469999e-02
            7.2779904913934002e-01 6.4280972920247004e-01 2.5426374419469999e-02
            8.3361584135278999e-01 -1.6805241863048001e-01 5.5935809884330001e-02
            8.3361584135278999e-01 1.6805241863048001e-01 5.5935809884330001e-02
            9.0714987061330998e-01 -3.4637395417630001e-01 2.6063184348280002e-02
            9.0714987061330998e-01 3.4637395417630001e-01 2.6063184348280002e-02
            9.7102821992230004e-01 0.0000000000000000e+00 2.6282427566290002e-02];

end









function XYW=cub_unit_disk_cools_kim(deg)

%--------------------------------------------------------------------------
% Reference paper:
% R. Cools K.J. Kim
% A survey of known and cubature formulas for the unit disk
% Korean J. Comput. & Appl. Math. Vol. 7(2000), No. 3, pp. 477 - 485.
%--------------------------------------------------------------------------
% | Deg | Card | type |
% .....................
% | 17  |  57  |  PI  |
% | 19  |  69  |  PO  |
% | 21  |  88  |  PO  |
%--------------------------------------------------------------------------

switch deg
    case 17

        % deg: 17
        % moment matching: 1.1e-15
        % cardinality:  57
        % max norm point: 9.77e-01

        XYW =[                  0   0.000000000000000 1.149833417999857e-01
            8.869676631639372e-01   0.000000000000000 4.266628153938678e-02
            -8.869676631639372e-01   0.000000000000000 4.266628153938678e-02
            0     8.869676631639372e-01     4.266628153938678e-02
            0    -8.869676631639372e-01     4.266628153938678e-02
            4.246339037432337e-01   0.000000000000000 8.793832535714553e-02
            -4.246339037432337e-01   0.000000000000000 8.793832535714553e-02
            0     4.246339037432337e-01     8.793832535714553e-02
            0    -4.246339037432337e-01     8.793832535714553e-02
            6.944690230808345e-01   0.000000000000000 7.620657046179324e-02
            -6.944690230808345e-01   0.000000000000000 7.620657046179324e-02
            0     6.944690230808345e-01     7.620657046179324e-02
            0    -6.944690230808345e-01     7.620657046179324e-02
            6.878535408269927e-01     6.878535408269927e-01     1.915652221885552e-02
            -6.878535408269927e-01     6.878535408269927e-01     1.915652221885552e-02
            6.878535408269927e-01    -6.878535408269927e-01     1.915652221885552e-02
            -6.878535408269927e-01    -6.878535408269927e-01     1.915652221885552e-02
            5.966476778145571e-01     5.966476778145571e-01     6.208572227313924e-02
            -5.966476778145571e-01     5.966476778145571e-01     6.208572227313924e-02
            5.966476778145571e-01    -5.966476778145571e-01     6.208572227313924e-02
            -5.966476778145571e-01    -5.966476778145571e-01     6.208572227313924e-02
            2.356225209153083e-01     2.356225209153083e-01     9.566496282041811e-02
            -2.356225209153083e-01     2.356225209153083e-01     9.566496282041811e-02
            2.356225209153083e-01    -2.356225209153083e-01     9.566496282041811e-02
            -2.356225209153083e-01    -2.356225209153083e-01     9.566496282041811e-02
            3.129475488834399e-01     5.489402552370146e-01     8.516253360428874e-02
            -3.129475488834399e-01     5.489402552370146e-01     8.516253360428874e-02
            3.129475488834399e-01    -5.489402552370146e-01     8.516253360428874e-02
            -3.129475488834399e-01    -5.489402552370146e-01     8.516253360428874e-02
            5.489402552370146e-01     3.129475488834399e-01     8.516253360428874e-02
            -5.489402552370146e-01     3.129475488834399e-01     8.516253360428874e-02
            5.489402552370146e-01    -3.129475488834399e-01     8.516253360428874e-02
            -5.489402552370146e-01    -3.129475488834399e-01     8.516253360428874e-02
            9.612122850461786e-01     1.738574508868360e-01     2.020123798956546e-02
            -9.612122850461786e-01     1.738574508868360e-01     2.020123798956546e-02
            9.612122850461786e-01    -1.738574508868360e-01     2.020123798956546e-02
            -9.612122850461786e-01    -1.738574508868360e-01     2.020123798956546e-02
            1.738574508868360e-01     9.612122850461786e-01     2.020123798956546e-02
            -1.738574508868360e-01     9.612122850461786e-01     2.020123798956546e-02
            1.738574508868360e-01    -9.612122850461786e-01     2.020123798956546e-02
            -1.738574508868360e-01    -9.612122850461786e-01     2.020123798956546e-02
            3.053873222521473e-01     7.903548753114861e-01     5.683457171315697e-02
            -3.053873222521473e-01     7.903548753114861e-01     5.683457171315697e-02
            3.053873222521473e-01    -7.903548753114861e-01     5.683457171315697e-02
            -3.053873222521473e-01    -7.903548753114861e-01     5.683457171315697e-02
            7.903548753114861e-01     3.053873222521473e-01     5.683457171315697e-02
            -7.903548753114861e-01     3.053873222521473e-01     5.683457171315697e-02
            7.903548753114861e-01    -3.053873222521473e-01     5.683457171315697e-02
            -7.903548753114861e-01    -3.053873222521473e-01     5.683457171315697e-02
            8.493729040963280e-01     4.627005659829375e-01     2.426862833134554e-02
            -8.493729040963280e-01     4.627005659829375e-01     2.426862833134554e-02
            8.493729040963280e-01    -4.627005659829375e-01     2.426862833134554e-02
            -8.493729040963280e-01    -4.627005659829375e-01     2.426862833134554e-02
            4.627005659829375e-01     8.493729040963280e-01     2.426862833134554e-02
            -4.627005659829375e-01     8.493729040963280e-01     2.426862833134554e-02
            4.627005659829375e-01    -8.493729040963280e-01     2.426862833134554e-02
            -4.627005659829375e-01    -8.493729040963280e-01     2.426862833134554e-02];

    case 19

        % deg: 19
        % moment matching: 1.9e-15
        % cardinality:  69
        % max norm point: 1.02e+00

        XYW=[   0.000000000000000                     0.000000000000000     9.282618274172910e-02
            7.983661095793183e-01   0.000000000000000 6.121064971417163e-02
            -7.983661095793183e-01   0.000000000000000 6.121064971417163e-02
            0     7.983661095793183e-01     6.121064971417163e-02
            0    -7.983661095793183e-01     6.121064971417163e-02
            3.004229233677078e-01   0.000000000000000 8.470130507409263e-02
            -3.004229233677078e-01   0.000000000000000 8.470130507409263e-02
            0     3.004229233677078e-01     8.470130507409263e-02
            0    -3.004229233677078e-01     8.470130507409263e-02
            9.961533343863167e-01   0.000000000000000 6.491852974192266e-03
            -9.961533343863167e-01   0.000000000000000 6.491852974192266e-03
            0     9.961533343863167e-01     6.491852974192266e-03
            0    -9.961533343863167e-01     6.491852974192266e-03
            6.958319163744974e-01     6.958319163744974e-01     1.410434731711371e-02
            -6.958319163744974e-01     6.958319163744974e-01     1.410434731711371e-02
            6.958319163744974e-01    -6.958319163744974e-01     1.410434731711371e-02
            -6.958319163744974e-01    -6.958319163744974e-01     1.410434731711371e-02
            2.867039116823771e-01     2.867039116823771e-01     7.759093834919630e-02
            -2.867039116823771e-01     2.867039116823771e-01     7.759093834919630e-02
            2.867039116823771e-01    -2.867039116823771e-01     7.759093834919630e-02
            -2.867039116823771e-01    -2.867039116823771e-01     7.759093834919630e-02
            6.166787816358777e-01     6.166787816358777e-01     4.684192284883144e-02
            -6.166787816358777e-01     6.166787816358777e-01     4.684192284883144e-02
            6.166787816358777e-01    -6.166787816358777e-01     4.684192284883144e-02
            -6.166787816358777e-01    -6.166787816358777e-01     4.684192284883144e-02
            4.721421263753088e-01     4.721421263753088e-01     7.152345880301066e-02
            -4.721421263753088e-01     4.721421263753088e-01     7.152345880301066e-02
            4.721421263753088e-01    -4.721421263753088e-01     7.152345880301066e-02
            -4.721421263753088e-01    -4.721421263753088e-01     7.152345880301066e-02
            5.668608215785862e-01     1.558778337876224e-01     7.726777790542050e-02
            -5.668608215785862e-01     1.558778337876224e-01     7.726777790542050e-02
            5.668608215785862e-01    -1.558778337876224e-01     7.726777790542050e-02
            -5.668608215785862e-01    -1.558778337876224e-01     7.726777790542050e-02
            1.558778337876224e-01     5.668608215785862e-01     7.726777790542050e-02
            -1.558778337876224e-01     5.668608215785862e-01     7.726777790542050e-02
            1.558778337876224e-01    -5.668608215785862e-01     7.726777790542050e-02
            -1.558778337876224e-01    -5.668608215785862e-01     7.726777790542050e-02
            9.255512731085483e-01     1.734769547117253e-01     3.435058365789666e-02
            -9.255512731085483e-01     1.734769547117253e-01     3.435058365789666e-02
            9.255512731085483e-01    -1.734769547117253e-01     3.435058365789666e-02
            -9.255512731085483e-01    -1.734769547117253e-01     3.435058365789666e-02
            1.734769547117253e-01     9.255512731085483e-01     3.435058365789666e-02
            -1.734769547117253e-01     9.255512731085483e-01     3.435058365789666e-02
            1.734769547117253e-01    -9.255512731085483e-01     3.435058365789666e-02
            -1.734769547117253e-01    -9.255512731085483e-01     3.435058365789666e-02
            8.382214435505526e-01     4.602599656372460e-01     2.836894938451606e-02
            -8.382214435505526e-01     4.602599656372460e-01     2.836894938451606e-02
            8.382214435505526e-01    -4.602599656372460e-01     2.836894938451606e-02
            -8.382214435505526e-01    -4.602599656372460e-01     2.836894938451606e-02
            4.602599656372460e-01     8.382214435505526e-01     2.836894938451606e-02
            -4.602599656372460e-01     8.382214435505526e-01     2.836894938451606e-02
            4.602599656372460e-01    -8.382214435505526e-01     2.836894938451606e-02
            -4.602599656372460e-01    -8.382214435505526e-01     2.836894938451606e-02
            9.667087658004858e-01     3.373792590174047e-01     1.866907432075110e-03
            -9.667087658004858e-01     3.373792590174047e-01     1.866907432075110e-03
            9.667087658004858e-01    -3.373792590174047e-01     1.866907432075110e-03
            -9.667087658004858e-01    -3.373792590174047e-01     1.866907432075110e-03
            3.373792590174047e-01     9.667087658004858e-01     1.866907432075110e-03
            -3.373792590174047e-01     9.667087658004858e-01     1.866907432075110e-03
            3.373792590174047e-01    -9.667087658004858e-01     1.866907432075110e-03
            -3.373792590174047e-01    -9.667087658004858e-01     1.866907432075110e-03
            7.377339001238997e-01     3.268524087084572e-01     5.800935293579536e-02
            -7.377339001238997e-01     3.268524087084572e-01     5.800935293579536e-02
            7.377339001238997e-01    -3.268524087084572e-01     5.800935293579536e-02
            -7.377339001238997e-01    -3.268524087084572e-01     5.800935293579536e-02
            3.268524087084572e-01     7.377339001238997e-01     5.800935293579536e-02
            -3.268524087084572e-01     7.377339001238997e-01     5.800935293579536e-02
            3.268524087084572e-01    -7.377339001238997e-01     5.800935293579536e-02
            -3.268524087084572e-01    -7.377339001238997e-01     5.800935293579536e-02];

    case 21

        % deg: 21
        % moment matching: 3.2e-15
        % cardinality:  88
        % max norm point: 1.09e+00

        XYW=[1.808996367091443e-01   0.000000000000000 6.471208815623311e-02
            -1.808996367091443e-01   0.000000000000000 6.471208815623311e-02
            0     1.808996367091443e-01     6.471208815623311e-02
            0    -1.808996367091443e-01     6.471208815623311e-02
            9.839153771475543e-01   0.000000000000000 1.220325734607774e-02
            -9.839153771475543e-01   0.000000000000000 1.220325734607774e-02
            0     9.839153771475543e-01     1.220325734607774e-02
            0    -9.839153771475543e-01     1.220325734607774e-02
            7.736438961473781e-01     7.736438961473781e-01     4.621446676487613e-05
            -7.736438961473781e-01     7.736438961473781e-01     4.621446676487613e-05
            7.736438961473781e-01    -7.736438961473781e-01     4.621446676487613e-05
            -7.736438961473781e-01    -7.736438961473781e-01     4.621446676487613e-05
            6.797593713482361e-01     6.797593713482361e-01     2.094112367408499e-02
            -6.797593713482361e-01     6.797593713482361e-01     2.094112367408499e-02
            6.797593713482361e-01    -6.797593713482361e-01     2.094112367408499e-02
            -6.797593713482361e-01    -6.797593713482361e-01     2.094112367408499e-02
            5.923056099556108e-01     5.923056099556108e-01     4.432176695412252e-02
            -5.923056099556108e-01     5.923056099556108e-01     4.432176695412252e-02
            5.923056099556108e-01    -5.923056099556108e-01     4.432176695412252e-02
            -5.923056099556108e-01    -5.923056099556108e-01     4.432176695412252e-02
            4.177444474035338e-01     4.177444474035338e-01     5.271569114926970e-02
            -4.177444474035338e-01     4.177444474035338e-01     5.271569114926970e-02
            4.177444474035338e-01    -4.177444474035338e-01     5.271569114926970e-02
            -4.177444474035338e-01    -4.177444474035338e-01     5.271569114926970e-02
            8.978928949520858e-01     1.337516952659082e-01     2.869002584930492e-02
            -8.978928949520858e-01     1.337516952659082e-01     2.869002584930492e-02
            8.978928949520858e-01    -1.337516952659082e-01     2.869002584930492e-02
            -8.978928949520858e-01    -1.337516952659082e-01     2.869002584930492e-02
            1.337516952659082e-01     8.978928949520858e-01     2.869002584930492e-02
            -1.337516952659082e-01     8.978928949520858e-01     2.869002584930492e-02
            1.337516952659082e-01    -8.978928949520858e-01     2.869002584930492e-02
            -1.337516952659082e-01    -8.978928949520858e-01     2.869002584930492e-02
            9.392735248696531e-01     2.798573269647498e-01     1.381970934464873e-02
            -9.392735248696531e-01     2.798573269647498e-01     1.381970934464873e-02
            9.392735248696531e-01    -2.798573269647498e-01     1.381970934464873e-02
            -9.392735248696531e-01    -2.798573269647498e-01     1.381970934464873e-02
            2.798573269647498e-01     9.392735248696531e-01     1.381970934464873e-02
            -2.798573269647498e-01     9.392735248696531e-01     1.381970934464873e-02
            2.798573269647498e-01    -9.392735248696531e-01     1.381970934464873e-02
            -2.798573269647498e-01    -9.392735248696531e-01     1.381970934464873e-02
            7.909967138562572e-01     3.938557657395190e-01     3.703488820285528e-02
            -7.909967138562572e-01     3.938557657395190e-01     3.703488820285528e-02
            7.909967138562572e-01    -3.938557657395190e-01     3.703488820285528e-02
            -7.909967138562572e-01    -3.938557657395190e-01     3.703488820285528e-02
            3.938557657395190e-01     7.909967138562572e-01     3.703488820285528e-02
            -3.938557657395190e-01     7.909967138562572e-01     3.703488820285528e-02
            3.938557657395190e-01    -7.909967138562572e-01     3.703488820285528e-02
            -3.938557657395190e-01    -7.909967138562572e-01     3.703488820285528e-02
            5.660548144579249e-01     1.088500580678623e-01     4.752013660747483e-02
            -5.660548144579249e-01     1.088500580678623e-01     4.752013660747483e-02
            5.660548144579249e-01    -1.088500580678623e-01     4.752013660747483e-02
            -5.660548144579249e-01    -1.088500580678623e-01     4.752013660747483e-02
            1.088500580678623e-01     5.660548144579249e-01     4.752013660747483e-02
            -1.088500580678623e-01     5.660548144579249e-01     4.752013660747483e-02
            1.088500580678623e-01    -5.660548144579249e-01     4.752013660747483e-02
            -1.088500580678623e-01    -5.660548144579249e-01     4.752013660747483e-02
            8.318077646484591e-01     5.122407254156560e-01     1.429616512013661e-02
            -8.318077646484591e-01     5.122407254156560e-01     1.429616512013661e-02
            8.318077646484591e-01    -5.122407254156560e-01     1.429616512013661e-02
            -8.318077646484591e-01    -5.122407254156560e-01     1.429616512013661e-02
            5.122407254156560e-01     8.318077646484591e-01     1.429616512013661e-02
            -5.122407254156560e-01     8.318077646484591e-01     1.429616512013661e-02
            5.122407254156560e-01    -8.318077646484591e-01     1.429616512013661e-02
            -5.122407254156560e-01    -8.318077646484591e-01     1.429616512013661e-02
            7.601058306926632e-01     1.283059541548807e-01     4.292376366988895e-02
            -7.601058306926632e-01     1.283059541548807e-01     4.292376366988895e-02
            7.601058306926632e-01    -1.283059541548807e-01     4.292376366988895e-02
            -7.601058306926632e-01    -1.283059541548807e-01     4.292376366988895e-02
            1.283059541548807e-01     7.601058306926632e-01     4.292376366988895e-02
            -1.283059541548807e-01     7.601058306926632e-01     4.292376366988895e-02
            1.283059541548807e-01    -7.601058306926632e-01     4.292376366988895e-02
            -1.283059541548807e-01    -7.601058306926632e-01     4.292376366988895e-02
            6.220933981210879e-01     3.469104071984239e-01     4.744409981166994e-02
            -6.220933981210879e-01     3.469104071984239e-01     4.744409981166994e-02
            6.220933981210879e-01    -3.469104071984239e-01     4.744409981166994e-02
            -6.220933981210879e-01    -3.469104071984239e-01     4.744409981166994e-02
            3.469104071984239e-01     6.220933981210879e-01     4.744409981166994e-02
            -3.469104071984239e-01     6.220933981210879e-01     4.744409981166994e-02
            3.469104071984239e-01    -6.220933981210879e-01     4.744409981166994e-02
            -3.469104071984239e-01    -6.220933981210879e-01     4.744409981166994e-02
            3.620005927676854e-01     1.667619187722297e-01     6.350022221946844e-02
            -3.620005927676854e-01     1.667619187722297e-01     6.350022221946844e-02
            3.620005927676854e-01    -1.667619187722297e-01     6.350022221946844e-02
            -3.620005927676854e-01    -1.667619187722297e-01     6.350022221946844e-02
            1.667619187722297e-01     3.620005927676854e-01     6.350022221946844e-02
            -1.667619187722297e-01     3.620005927676854e-01     6.350022221946844e-02
            1.667619187722297e-01    -3.620005927676854e-01     6.350022221946844e-02
            -1.667619187722297e-01    -3.620005927676854e-01     6.350022221946844e-02];

end


















function XYW=cub_unit_disk_cools_hagemans85(deg)

%--------------------------------------------------------------------------
% Reference paper:
% R. Cools, A. Haegemans,
% Construction of fully symmetric cubature formulae of degree 4k-3 for fully
% symmetric planar regions
% 1985, Report TW 71, Dept. of Computer Science, KU Leuven,
%--------------------------------------------------------------------------
% | Deg | Card | type |
% ......................
% | 5   |  8   |  PO  |
% | 9   |  20  |  PO  |
%--------------------------------------------------------------------------

switch deg
    case 5

        % deg:  5
        % moment matching: 1.6e-12
        % cardinality:   8
        % max norm point: 1.26e+00

        XYW=[4.597008433810000e-01     4.597008433810000e-01     7.327864624924677e-01
            -4.597008433810000e-01     4.597008433810000e-01     7.327864624924677e-01
            4.597008433810000e-01    -4.597008433810000e-01     7.327864624924677e-01
            -4.597008433810000e-01    -4.597008433810000e-01     7.327864624924677e-01
            1.255926060400000e+00   0.000000000000000 5.261170090466640e-02
            -1.255926060400000e+00   0.000000000000000 5.261170090466640e-02
            0.000000000000000     1.255926060400000e+00     5.261170090466640e-02
            0.000000000000000    -1.255926060400000e+00     5.261170090466640e-02];

    case 9


        % deg:  9
        % moment matching: 2.9e-12
        % cardinality:  20
        % max norm point: 1.01e+00

        XYW=[3.438553452940000e-01     9.447780171420000e-01     3.878223761159647e-02
            -3.438553452940000e-01     9.447780171420000e-01     3.878223761159647e-02
            3.438553452940000e-01    -9.447780171420000e-01     3.878223761159647e-02
            -3.438553452940000e-01    -9.447780171420000e-01     3.878223761159647e-02
            9.447780171420000e-01     3.438553452940000e-01     3.878223761159647e-02
            -9.447780171420000e-01     3.438553452940000e-01     3.878223761159647e-02
            9.447780171420000e-01    -3.438553452940000e-01     3.878223761159647e-02
            -9.447780171420000e-01    -3.438553452940000e-01     3.878223761159647e-02
            2.774965002970000e-01     2.774965002970000e-01     2.930225148632210e-01
            -2.774965002970000e-01     2.774965002970000e-01     2.930225148632210e-01
            2.774965002970000e-01    -2.774965002970000e-01     2.930225148632210e-01
            -2.774965002970000e-01    -2.774965002970000e-01     2.930225148632210e-01
            5.923553873960000e-01     5.923553873960000e-01     1.851958765247484e-01
            -5.923553873960000e-01     5.923553873960000e-01     1.851958765247484e-01
            5.923553873960000e-01    -5.923553873960000e-01     1.851958765247484e-01
            -5.923553873960000e-01    -5.923553873960000e-01     1.851958765247484e-01
            7.786108199230000e-01     0.000000000000000     2.296152967862859e-01
            -7.786108199230000e-01    0.000000000000000     2.296152967862859e-01
            0.000000000000000     7.786108199230000e-01     2.296152967862859e-01
            0.000000000000000    -7.786108199230000e-01     2.296152967862859e-01];
end









function XYW=cub_unit_disk_wissmann_becker(deg)

%--------------------------------------------------------------------------
% Reference paper:
%     Johannes W. Wissmann and Thomas Becker,
%     Partially Symmetric Cubature Formulas for Even Degrees of Exactness,
%     SIAM J. Numer. Anal., 23(3), 676?685
%--------------------------------------------------------------------------
% | Deg | Card | Qual |
% .......................
% | 6   |  10  |  PO  |
% | 6   |  11  |  PO  |
% | 8   |  16  |  PI  |
%--------------------------------------------------------------------------

switch deg
    case 6

        subcase=2;

        switch subcase
            case 1

                % deg:  6
                % moment matching: 8.7e-15
                % cardinality:  10
                % max norm point: 1.18e+00

                XYW=[                    0     9.148742615449320e-01     1.921063304091670e-01
                    0    -2.063555434122150e-01     6.428831724532350e-01
                    9.788390470642830e-01     6.629524536610440e-01     3.032781530534200e-02
                    -9.788390470642830e-01     6.629524536610440e-01     3.032781530534200e-02
                    4.582365157086840e-01     4.648898897832780e-01     5.023584088123140e-01
                    -4.582365157086840e-01     4.648898897832780e-01     5.023584088123140e-01
                    8.020170040723430e-01    -1.742073788395560e-01     3.405400945006880e-01
                    -8.020170040723430e-01    -1.742073788395560e-01     3.405400945006880e-01
                    3.736838643044990e-01    -7.707498531488070e-01     2.800752567453520e-01
                    -3.736838643044990e-01    -7.707498531488070e-01     2.800752567453520e-01];

            otherwise

                % deg:  6
                % moment matching: 3.9e-15
                % cardinality:  11
                % max norm point: 1.03e+00

                XYW=[                    0                         0     5.526875964648710e-01
                    0     1.032546063890842e+00     7.609043930254500e-02
                    0    -7.263598460429440e-01     4.416905721224400e-01
                    9.820096624382970e-01     3.190742812172300e-01     7.609043930254500e-02
                    -9.820096624382970e-01     3.190742812172300e-01     7.609043930254500e-02
                    6.069153486676780e-01    -8.353473131626510e-01     7.609043930254500e-02
                    -6.069153486676780e-01    -8.353473131626510e-01     7.609043930254500e-02
                    4.269436053614740e-01     5.876374594803120e-01     4.416905721224400e-01
                    -4.269436053614740e-01     5.876374594803120e-01     4.416905721224400e-01
                    6.908092647542869e-01    -2.244575364588400e-01     4.416905721224400e-01
                    -6.908092647542869e-01    -2.244575364588400e-01     4.416905721224400e-01];
        end

    case 8

        % deg:  8
        % moment matching: 3.2e-15
        % cardinality:  16
        % max norm point: 9.53e-01

        XYW=[                    0                         0     3.345214399655800e-01
            0     9.533211755218070e-01     8.791138770463899e-02
            0    -8.824580989001261e-01     1.676280391064690e-01
            0     5.823341881204990e-01     3.058748159137350e-01
            9.066623161021710e-01     2.945924443337410e-01     8.791138770463899e-02
            -9.066623161021710e-01     2.945924443337410e-01     8.791138770463899e-02
            5.603481276698430e-01    -7.712530320946440e-01     8.791138770463899e-02
            -5.603481276698430e-01    -7.712530320946440e-01     8.791138770463899e-02
            5.186958562995470e-01     7.139235988340100e-01     1.676280391064690e-01
            -5.186958562995470e-01     7.139235988340100e-01     1.676280391064690e-01
            8.392675253163980e-01    -2.726945493839470e-01     1.676280391064690e-01
            -8.392675253163980e-01    -2.726945493839470e-01     1.676280391064690e-01
            5.538327242734490e-01     1.799511605347720e-01     3.058748159137350e-01
            -5.538327242734490e-01     1.799511605347720e-01     3.058748159137350e-01
            3.422874476829400e-01    -4.711182545950220e-01     3.058748159137350e-01
            -3.422874476829400e-01    -4.711182545950220e-01     3.058748159137350e-01];
end









function XYW=cub_unit_disk_rabinowitz_richter(deg)

%--------------------------------------------------------------------------
% Reference paper:
%     Philip Rabinowitz and Nira Richter,
%     Perfectly Symmetric Two-Dimensional Integration Formulas with Minimal
%     Numbers of Points,
%     Mathematics of Computation, Vol. 23, No. 108 (Oct., 1969), pp. 765-779,
%--------------------------------------------------------------------------
% | Deg | Card | Qual |
% .......................
% |  9  |  20  |  PO  |
% |  9  |  21  |  PO  |
% | 11  |  28  |  PI  |
% | 11  |  28  |  PI  |
% | 13  |  37  |  PI  |
% | 15  |  44  |  PI  |
%--------------------------------------------------------------------------

switch deg
    case 9

        subcase=2;
        switch subcase
            case 1
                % moment matching: 9.7e-16
                % cardinality:  20
                % max norm point: 1.01e+00
                XYW=[     8.377170225998396e-01                         0     1.851958765246450e-01
                    -8.377170225998396e-01                         0     1.851958765246450e-01
                    0     8.377170225998396e-01     1.851958765246450e-01
                    0    -8.377170225998396e-01     1.851958765246450e-01
                    3.924393142315810e-01                         0     2.930225148631698e-01
                    -3.924393142315810e-01                         0     2.930225148631698e-01
                    0     3.924393142315810e-01     2.930225148631698e-01
                    0    -3.924393142315810e-01     2.930225148631698e-01
                    5.505609906724360e-01     5.505609906724360e-01     2.296152967863584e-01
                    -5.505609906724360e-01     5.505609906724360e-01     2.296152967863584e-01
                    5.505609906724360e-01    -5.505609906724360e-01     2.296152967863584e-01
                    -5.505609906724360e-01    -5.505609906724360e-01     2.296152967863584e-01
                    4.249164962326038e-01     9.112013890413142e-01     3.878223761163760e-02
                    -4.249164962326038e-01     9.112013890413142e-01     3.878223761163760e-02
                    4.249164962326038e-01    -9.112013890413142e-01     3.878223761163760e-02
                    -4.249164962326038e-01    -9.112013890413142e-01     3.878223761163760e-02
                    9.112013890413142e-01     4.249164962326038e-01     3.878223761163760e-02
                    -9.112013890413142e-01     4.249164962326038e-01     3.878223761163760e-02
                    9.112013890413142e-01    -4.249164962326038e-01     3.878223761163760e-02
                    -9.112013890413142e-01    -4.249164962326038e-01     3.878223761163760e-02];

            case 2
                % deg:  9
                % moment matching: 7.9e-15
                % cardinality:  21
                % max norm point: 1.20e+00
                XYW=[
                    0                         0     4.317395418843000e-03
                    9.499490053854548e-01                         0     7.348670163034730e-02
                    -9.499490053854548e-01                         0     7.348670163034730e-02
                    0     9.499490053854548e-01     7.348670163034730e-02
                    0    -9.499490053854548e-01     7.348670163034730e-02
                    4.184300297249359e-01                         0     3.295210136662689e-01
                    -4.184300297249359e-01                         0     3.295210136662689e-01
                    0     4.184300297249359e-01     3.295210136662689e-01
                    0    -4.184300297249359e-01     3.295210136662689e-01
                    8.485281374238570e-01     8.485281374238570e-01     4.609139996675700e-03
                    -8.485281374238570e-01     8.485281374238570e-01     4.609139996675700e-03
                    8.485281374238570e-01    -8.485281374238570e-01     4.609139996675700e-03
                    -8.485281374238570e-01    -8.485281374238570e-01     4.609139996675700e-03
                    3.830079234911947e-01     7.409163950514299e-01     1.883509796247228e-01
                    -3.830079234911947e-01     7.409163950514299e-01     1.883509796247228e-01
                    3.830079234911947e-01    -7.409163950514299e-01     1.883509796247228e-01
                    -3.830079234911947e-01    -7.409163950514299e-01     1.883509796247228e-01
                    7.409163950514299e-01     3.830079234911947e-01     1.883509796247228e-01
                    -7.409163950514299e-01     3.830079234911947e-01     1.883509796247228e-01
                    7.409163950514299e-01    -3.830079234911947e-01     1.883509796247228e-01
                    -7.409163950514299e-01    -3.830079234911947e-01     1.883509796247228e-01];

        end


    case 11

        subcase=2;

        switch subcase
            case 1

                % deg: 11
                % moment matching: 1.5e-15
                % cardinality:  28
                % max norm point: 9.62e-01

                XYW=[     9.619017737816972e-01                         0     5.131005271235660e-02
                    -9.619017737816972e-01                         0     5.131005271235660e-02
                    0     9.619017737816972e-01     5.131005271235660e-02
                    0    -9.619017737816972e-01     5.131005271235660e-02
                    7.745966692414835e-01                         0     1.136282065100473e-01
                    -7.745966692414835e-01                         0     1.136282065100473e-01
                    0     7.745966692414835e-01     1.136282065100473e-01
                    0    -7.745966692414835e-01     1.136282065100473e-01
                    3.287526591967855e-01                         0     2.083682752319387e-01
                    -3.287526591967855e-01                         0     2.083682752319387e-01
                    0     3.287526591967855e-01     2.083682752319387e-01
                    0    -3.287526591967855e-01     2.083682752319387e-01
                    4.683708939890903e-01     8.112421851755608e-01     7.906797796832821e-02
                    -4.683708939890903e-01     8.112421851755608e-01     7.906797796832821e-02
                    4.683708939890903e-01    -8.112421851755608e-01     7.906797796832821e-02
                    -4.683708939890903e-01    -8.112421851755608e-01     7.906797796832821e-02
                    8.112421851755608e-01     4.683708939890903e-01     7.906797796832821e-02
                    -8.112421851755608e-01     4.683708939890903e-01     7.906797796832821e-02
                    8.112421851755608e-01    -4.683708939890903e-01     7.906797796832821e-02
                    -8.112421851755608e-01    -4.683708939890903e-01     7.906797796832821e-02
                    3.375826402485671e-01     5.847102846637651e-01     1.269778365032246e-01
                    -3.375826402485671e-01     5.847102846637651e-01     1.269778365032246e-01
                    3.375826402485671e-01    -5.847102846637651e-01     1.269778365032246e-01
                    -3.375826402485671e-01    -5.847102846637651e-01     1.269778365032246e-01
                    5.847102846637651e-01     3.375826402485671e-01     1.269778365032246e-01
                    -5.847102846637651e-01     3.375826402485671e-01     1.269778365032246e-01
                    5.847102846637651e-01    -3.375826402485671e-01     1.269778365032246e-01
                    -5.847102846637651e-01    -3.375826402485671e-01     1.269778365032246e-01];

            case 2

                % deg: 11
                % moment matching: 1.4e-15
                % cardinality:  28
                % max norm point: 9.95e-01
                XYW=[     9.669004345445009e-01                         0     4.783963264042470e-02
                    -9.669004345445009e-01                         0     4.783963264042470e-02
                    0     9.669004345445009e-01     4.783963264042470e-02
                    0    -9.669004345445009e-01     4.783963264042470e-02
                    7.226054070052285e-01                         0     1.597003917456590e-01
                    -7.226054070052285e-01                         0     1.597003917456590e-01
                    0     7.226054070052285e-01     1.597003917456590e-01
                    0    -7.226054070052285e-01     1.597003917456590e-01
                    3.233163607428629e-01                         0     2.016322022034297e-01
                    -3.233163607428629e-01                         0     2.016322022034297e-01
                    0     3.233163607428629e-01     2.016322022034297e-01
                    0    -3.233163607428629e-01     2.016322022034297e-01
                    7.036534680827587e-01     7.036534680827587e-01     1.650897337836640e-02
                    -7.036534680827587e-01     7.036534680827587e-01     1.650897337836640e-02
                    7.036534680827587e-01    -7.036534680827587e-01     1.650897337836640e-02
                    -7.036534680827587e-01    -7.036534680827587e-01     1.650897337836640e-02
                    4.638891735186042e-01     4.638891735186042e-01     1.801837855454157e-01
                    -4.638891735186042e-01     4.638891735186042e-01     1.801837855454157e-01
                    4.638891735186042e-01    -4.638891735186042e-01     1.801837855454157e-01
                    -4.638891735186042e-01    -4.638891735186042e-01     1.801837855454157e-01
                    4.135214625627066e-01     8.138386408455507e-01     8.976658894207649e-02
                    -4.135214625627066e-01     8.138386408455507e-01     8.976658894207649e-02
                    4.135214625627066e-01    -8.138386408455507e-01     8.976658894207649e-02
                    -4.135214625627066e-01    -8.138386408455507e-01     8.976658894207649e-02
                    8.138386408455507e-01     4.135214625627066e-01     8.976658894207649e-02
                    -8.138386408455507e-01     4.135214625627066e-01     8.976658894207649e-02
                    8.138386408455507e-01    -4.135214625627066e-01     8.976658894207649e-02
                    -8.138386408455507e-01    -4.135214625627066e-01     8.976658894207649e-02];

        end

    case 13

        % deg: 13
        % moment matching: 1.7e-15
        % cardinality:  37
        % max norm point: 9.71e-01

        XYW=[                         0                         0     1.604310638138027e-01
            3.879803784555729e-01                         0     1.424323658922069e-01
            -3.879803784555729e-01                         0     1.424323658922069e-01
            0     3.879803784555729e-01     1.424323658922069e-01
            0    -3.879803784555729e-01     1.424323658922069e-01
            9.358527527678654e-01                         0     5.568453910709620e-02
            -9.358527527678654e-01                         0     5.568453910709620e-02
            0     9.358527527678654e-01     5.568453910709620e-02
            0    -9.358527527678654e-01     5.568453910709620e-02
            7.134059509780893e-01                         0     1.114444717392537e-01
            -7.134059509780893e-01                         0     1.114444717392537e-01
            0     7.134059509780893e-01     1.114444717392537e-01
            0    -7.134059509780893e-01     1.114444717392537e-01
            6.759153919798939e-01     6.759153919798939e-01     4.497899468266130e-02
            -6.759153919798939e-01     6.759153919798939e-01     4.497899468266130e-02
            6.759153919798939e-01    -6.759153919798939e-01     4.497899468266130e-02
            -6.759153919798939e-01    -6.759153919798939e-01     4.497899468266130e-02
            3.835039628013994e-01     3.835039628013994e-01     1.347199228191621e-01
            -3.835039628013994e-01     3.835039628013994e-01     1.347199228191621e-01
            3.835039628013994e-01    -3.835039628013994e-01     1.347199228191621e-01
            -3.835039628013994e-01    -3.835039628013994e-01     1.347199228191621e-01
            3.464101615137754e-01     9.066277008560241e-01     3.166188264167740e-02
            -3.464101615137754e-01     9.066277008560241e-01     3.166188264167740e-02
            3.464101615137754e-01    -9.066277008560241e-01     3.166188264167740e-02
            -3.464101615137754e-01    -9.066277008560241e-01     3.166188264167740e-02
            9.066277008560241e-01     3.464101615137754e-01     3.166188264167740e-02
            -9.066277008560241e-01     3.464101615137754e-01     3.166188264167740e-02
            9.066277008560241e-01    -3.464101615137754e-01     3.166188264167740e-02
            -9.066277008560241e-01    -3.464101615137754e-01     3.166188264167740e-02
            7.106593341863341e-01     3.816598192059473e-01     9.635316896013130e-02
            -7.106593341863341e-01     3.816598192059473e-01     9.635316896013130e-02
            7.106593341863341e-01    -3.816598192059473e-01     9.635316896013130e-02
            -7.106593341863341e-01    -3.816598192059473e-01     9.635316896013130e-02
            3.816598192059473e-01     7.106593341863341e-01     9.635316896013130e-02
            -3.816598192059473e-01     7.106593341863341e-01     9.635316896013130e-02
            3.816598192059473e-01    -7.106593341863341e-01     9.635316896013130e-02
            -3.816598192059473e-01    -7.106593341863341e-01     9.635316896013130e-02];

    case 15

        % deg: 15
        % moment matching: 2.0e-15
        % cardinality:  44
        % max norm point: 9.90e-01

        XYW=[     2.528637970912295e-01                         0     1.252902085642858e-01
            -2.528637970912295e-01                         0     1.252902085642858e-01
            0     2.528637970912295e-01     1.252902085642858e-01
            0    -2.528637970912295e-01     1.252902085642858e-01
            5.777289284448234e-01                         0     1.095003911263660e-01
            -5.777289284448234e-01                         0     1.095003911263660e-01
            0     5.777289284448234e-01     1.095003911263660e-01
            0    -5.777289284448234e-01     1.095003911263660e-01
            9.897468025114907e-01                         0     1.671262549704350e-02
            -9.897468025114907e-01                         0     1.671262549704350e-02
            0     9.897468025114907e-01     1.671262549704350e-02
            0    -9.897468025114907e-01     1.671262549704350e-02
            8.738369566448817e-01                         0     6.623745579637630e-02
            -8.738369566448817e-01                         0     6.623745579637630e-02
            0     8.738369566448817e-01     6.623745579637630e-02
            0    -8.738369566448817e-01     6.623745579637630e-02
            3.754168246261542e-01     3.754168246261542e-01     1.274283726817204e-01
            -3.754168246261542e-01     3.754168246261542e-01     1.274283726817204e-01
            3.754168246261542e-01    -3.754168246261542e-01     1.274283726817204e-01
            -3.754168246261542e-01    -3.754168246261542e-01     1.274283726817204e-01
            6.892993807911362e-01     6.892993807911362e-01     2.610286018436050e-02
            -6.892993807911362e-01     6.892993807911362e-01     2.610286018436050e-02
            6.892993807911362e-01    -6.892993807911362e-01     2.610286018436050e-02
            -6.892993807911362e-01    -6.892993807911362e-01     2.610286018436050e-02
            5.976143046671968e-01     5.976143046671968e-01     6.600093466110460e-02
            -5.976143046671968e-01     5.976143046671968e-01     6.600093466110460e-02
            5.976143046671968e-01    -5.976143046671968e-01     6.600093466110460e-02
            -5.976143046671968e-01    -5.976143046671968e-01     6.600093466110460e-02
            3.657908004006625e-01     8.830971113185893e-01     4.252306582668240e-02
            -3.657908004006625e-01     8.830971113185893e-01     4.252306582668240e-02
            3.657908004006625e-01    -8.830971113185893e-01     4.252306582668240e-02
            -3.657908004006625e-01    -8.830971113185893e-01     4.252306582668240e-02
            8.830971113185893e-01     3.657908004006625e-01     4.252306582668240e-02
            -8.830971113185893e-01     3.657908004006625e-01     4.252306582668240e-02
            8.830971113185893e-01    -3.657908004006625e-01     4.252306582668240e-02
            -8.830971113185893e-01    -3.657908004006625e-01     4.252306582668240e-02
            2.930307227106603e-01     7.074387449600663e-01     8.153959161641321e-02
            -2.930307227106603e-01     7.074387449600663e-01     8.153959161641321e-02
            2.930307227106603e-01    -7.074387449600663e-01     8.153959161641321e-02
            -2.930307227106603e-01    -7.074387449600663e-01     8.153959161641321e-02
            7.074387449600663e-01     2.930307227106603e-01     8.153959161641321e-02
            -7.074387449600663e-01     2.930307227106603e-01     8.153959161641321e-02
            7.074387449600663e-01    -2.930307227106603e-01     8.153959161641321e-02
            -7.074387449600663e-01    -2.930307227106603e-01     8.153959161641321e-02];

end









function XYW=cub_rules_disk_2012(deg)

%--------------------------------------------------------------------------
% Other rules.
%--------------------------------------------------------------------------
% | Deg | Card | type|
% .....................
% |  1  |  1  |  PI  |
% |  2  |  4  |  PI  |
% |  3  |  4  |  PI  |
% |  4  |  8  |  PO  |
% |  5  |  8  |  PO  |
% |  6  | 12  |  PI  |
% |  7  | 12  |  PI  |
% |  8  | 20  |  PO  |
% |  9  | 20  |  PO  |
% | 12  | 36  |  PI  |
% | 13  | 36  |  PI  |
% | 14  | 44  |  PI  | *
% | 15  | 44  |  PI  | *
%--------------------------------------------------------------------------



switch deg

    case 1

        % moment matching: 8.9e-16
        % cardinality:   1
        % max norm point: 0.00e+00

        XYW=[
            0.000000000000000000e+00 0.000000000000000000e+00 3.141592653589793116e+00
            ];

    case 2

        % moment matching: 2.5e-15
        % cardinality:   4
        % max norm point: 7.07e-01

        XYW=[
            7.071067811865480168e-01 0.000000000000000000e+00 7.853981633974479459e-01
            -7.071067811865480168e-01 0.000000000000000000e+00 7.853981633974479459e-01
            0.000000000000000000e+00 7.071067811865480168e-01 7.853981633974479459e-01
            0.000000000000000000e+00 -7.071067811865480168e-01 7.853981633974479459e-01
            ];

    case   3

        % moment matching: 2.5e-15
        % cardinality:   4
        % max norm point: 7.07e-01

        XYW=[
            7.071067811865480168e-01 0.000000000000000000e+00 7.853981633974479459e-01
            -7.071067811865480168e-01 0.000000000000000000e+00 7.853981633974479459e-01
            0.000000000000000000e+00 7.071067811865480168e-01 7.853981633974479459e-01
            0.000000000000000000e+00 -7.071067811865480168e-01 7.853981633974479459e-01
            ];

    case   4

        % moment matching: 7.4e-15
        % cardinality:   8
        % max norm point: 1.26e+00

        XYW=[
            6.501151673437359779e-01 0.000000000000000000e+00 7.327864624926400117e-01
            -6.501151673437359779e-01 0.000000000000000000e+00 7.327864624926400117e-01
            0.000000000000000000e+00 6.501151673437359779e-01 7.327864624926400117e-01
            0.000000000000000000e+00 -6.501151673437359779e-01 7.327864624926400117e-01
            8.880738339771150347e-01 8.880738339771150347e-01 5.261170090480799671e-02
            -8.880738339771150347e-01 8.880738339771150347e-01 5.261170090480799671e-02
            8.880738339771150347e-01 -8.880738339771150347e-01 5.261170090480799671e-02
            -8.880738339771150347e-01 -8.880738339771150347e-01 5.261170090480799671e-02
            ];

    case   5

        % moment matching: 7.4e-15
        % cardinality:   8
        % max norm point: 1.26e+00

        XYW=[
            6.501151673437359779e-01 0.000000000000000000e+00 7.327864624926400117e-01
            -6.501151673437359779e-01 0.000000000000000000e+00 7.327864624926400117e-01
            0.000000000000000000e+00 6.501151673437359779e-01 7.327864624926400117e-01
            0.000000000000000000e+00 -6.501151673437359779e-01 7.327864624926400117e-01
            8.880738339771150347e-01 8.880738339771150347e-01 5.261170090480799671e-02
            -8.880738339771150347e-01 8.880738339771150347e-01 5.261170090480799671e-02
            8.880738339771150347e-01 -8.880738339771150347e-01 5.261170090480799671e-02
            -8.880738339771150347e-01 -8.880738339771150347e-01 5.261170090480799671e-02
            ];

    case   6

        % moment matching: 6.2e-15
        % cardinality:  12
        % max norm point: 9.11e-01

        XYW=[
            8.660254037844390407e-01 0.000000000000000000e+00 2.327105669325769954e-01
            -8.660254037844390407e-01 0.000000000000000000e+00 2.327105669325769954e-01
            0.000000000000000000e+00 8.660254037844390407e-01 2.327105669325769954e-01
            0.000000000000000000e+00 -8.660254037844390407e-01 2.327105669325769954e-01
            3.229149920673999863e-01 3.229149920673999863e-01 3.870777960062259937e-01
            -3.229149920673999863e-01 3.229149920673999863e-01 3.870777960062259937e-01
            3.229149920673999863e-01 -3.229149920673999863e-01 3.870777960062259937e-01
            -3.229149920673999863e-01 -3.229149920673999863e-01 3.870777960062259937e-01
            6.441713103894649572e-01 6.441713103894649572e-01 1.656098004586450123e-01
            -6.441713103894649572e-01 6.441713103894649572e-01 1.656098004586450123e-01
            6.441713103894649572e-01 -6.441713103894649572e-01 1.656098004586450123e-01
            -6.441713103894649572e-01 -6.441713103894649572e-01 1.656098004586450123e-01
            ];


    case   7

        % moment matching: 6.2e-15
        % cardinality:  12
        % max norm point: 9.11e-01

        XYW=[
            8.660254037844390407e-01 0.000000000000000000e+00 2.327105669325769954e-01
            -8.660254037844390407e-01 0.000000000000000000e+00 2.327105669325769954e-01
            0.000000000000000000e+00 8.660254037844390407e-01 2.327105669325769954e-01
            0.000000000000000000e+00 -8.660254037844390407e-01 2.327105669325769954e-01
            3.229149920673999863e-01 3.229149920673999863e-01 3.870777960062259937e-01
            -3.229149920673999863e-01 3.229149920673999863e-01 3.870777960062259937e-01
            3.229149920673999863e-01 -3.229149920673999863e-01 3.870777960062259937e-01
            -3.229149920673999863e-01 -3.229149920673999863e-01 3.870777960062259937e-01
            6.441713103894649572e-01 6.441713103894649572e-01 1.656098004586450123e-01
            -6.441713103894649572e-01 6.441713103894649572e-01 1.656098004586450123e-01
            6.441713103894649572e-01 -6.441713103894649572e-01 1.656098004586450123e-01
            -6.441713103894649572e-01 -6.441713103894649572e-01 1.656098004586450123e-01
            ];


    case   8

        % moment matching: 5.3e-14
        % cardinality:  20
        % max norm point: 1.18e+00

        XYW=[
            9.524588964344169462e-01 0.000000000000000000e+00 7.148882661739100475e-02
            -9.524588964344169462e-01 0.000000000000000000e+00 7.148882661739100475e-02
            0.000000000000000000e+00 9.524588964344169462e-01 7.148882661739100475e-02
            0.000000000000000000e+00 -9.524588964344169462e-01 7.148882661739100475e-02
            4.151876578787550010e-01 0.000000000000000000e+00 3.271768749281669941e-01
            -4.151876578787550010e-01 0.000000000000000000e+00 3.271768749281669941e-01
            0.000000000000000000e+00 4.151876578787550010e-01 3.271768749281669941e-01
            0.000000000000000000e+00 -4.151876578787550010e-01 3.271768749281669941e-01
            8.347949422162109778e-01 8.347949422162109778e-01 5.591341512851000055e-03
            -8.347949422162109778e-01 8.347949422162109778e-01 5.591341512851000055e-03
            8.347949422162109778e-01 -8.347949422162109778e-01 5.591341512851000055e-03
            -8.347949422162109778e-01 -8.347949422162109778e-01 5.591341512851000055e-03
            7.403344571735109847e-01 3.790169375308349853e-01 1.905705601695190099e-01
            -7.403344571735109847e-01 3.790169375308349853e-01 1.905705601695190099e-01
            7.403344571735109847e-01 -3.790169375308349853e-01 1.905705601695190099e-01
            -7.403344571735109847e-01 -3.790169375308349853e-01 1.905705601695190099e-01
            3.790169375308349853e-01 7.403344571735109847e-01 1.905705601695190099e-01
            -3.790169375308349853e-01 7.403344571735109847e-01 1.905705601695190099e-01
            3.790169375308349853e-01 -7.403344571735109847e-01 1.905705601695190099e-01
            -3.790169375308349853e-01 -7.403344571735109847e-01 1.905705601695190099e-01
            ];



    case   9

        % moment matching: 5.3e-14
        % cardinality:  20
        % max norm point: 1.18e+00

        XYW=[
            9.524588964344169462e-01 0.000000000000000000e+00 7.148882661739100475e-02
            -9.524588964344169462e-01 0.000000000000000000e+00 7.148882661739100475e-02
            0.000000000000000000e+00 9.524588964344169462e-01 7.148882661739100475e-02
            0.000000000000000000e+00 -9.524588964344169462e-01 7.148882661739100475e-02
            4.151876578787550010e-01 0.000000000000000000e+00 3.271768749281669941e-01
            -4.151876578787550010e-01 0.000000000000000000e+00 3.271768749281669941e-01
            0.000000000000000000e+00 4.151876578787550010e-01 3.271768749281669941e-01
            0.000000000000000000e+00 -4.151876578787550010e-01 3.271768749281669941e-01
            8.347949422162109778e-01 8.347949422162109778e-01 5.591341512851000055e-03
            -8.347949422162109778e-01 8.347949422162109778e-01 5.591341512851000055e-03
            8.347949422162109778e-01 -8.347949422162109778e-01 5.591341512851000055e-03
            -8.347949422162109778e-01 -8.347949422162109778e-01 5.591341512851000055e-03
            7.403344571735109847e-01 3.790169375308349853e-01 1.905705601695190099e-01
            -7.403344571735109847e-01 3.790169375308349853e-01 1.905705601695190099e-01
            7.403344571735109847e-01 -3.790169375308349853e-01 1.905705601695190099e-01
            -7.403344571735109847e-01 -3.790169375308349853e-01 1.905705601695190099e-01
            3.790169375308349853e-01 7.403344571735109847e-01 1.905705601695190099e-01
            -3.790169375308349853e-01 7.403344571735109847e-01 1.905705601695190099e-01
            3.790169375308349853e-01 -7.403344571735109847e-01 1.905705601695190099e-01
            -3.790169375308349853e-01 -7.403344571735109847e-01 1.905705601695190099e-01
            ];

    case  12

        % moment matching: 1.8e-13
        % cardinality:  36
        % max norm point: 9.97e-01

        XYW=[
            2.834028323488250178e-01 0.000000000000000000e+00 1.562398698373330119e-01
            -2.834028323488250178e-01 0.000000000000000000e+00 1.562398698373330119e-01
            0.000000000000000000e+00 2.834028323488250178e-01 1.562398698373330119e-01
            0.000000000000000000e+00 -2.834028323488250178e-01 1.562398698373330119e-01
            9.205754740362509514e-01 0.000000000000000000e+00 6.944740926697699768e-02
            -9.205754740362509514e-01 0.000000000000000000e+00 6.944740926697699768e-02
            0.000000000000000000e+00 9.205754740362509514e-01 6.944740926697699768e-02
            0.000000000000000000e+00 -9.205754740362509514e-01 6.944740926697699768e-02
            6.490072305780020256e-01 0.000000000000000000e+00 1.381459105348709915e-01
            -6.490072305780020256e-01 0.000000000000000000e+00 1.381459105348709915e-01
            0.000000000000000000e+00 6.490072305780020256e-01 1.381459105348709915e-01
            0.000000000000000000e+00 -6.490072305780020256e-01 1.381459105348709915e-01
            6.773551060280690317e-01 6.773551060280690317e-01 4.337658388715499858e-02
            -6.773551060280690317e-01 6.773551060280690317e-01 4.337658388715499858e-02
            6.773551060280690317e-01 -6.773551060280690317e-01 4.337658388715499858e-02
            -6.773551060280690317e-01 -6.773551060280690317e-01 4.337658388715499858e-02
            4.126722167632890237e-01 4.126722167632890237e-01 1.502283341400099947e-01
            -4.126722167632890237e-01 4.126722167632890237e-01 1.502283341400099947e-01
            4.126722167632890237e-01 -4.126722167632890237e-01 1.502283341400099947e-01
            -4.126722167632890237e-01 -4.126722167632890237e-01 1.502283341400099947e-01
            9.386945838351230531e-01 3.357676008295389924e-01 1.819932192611799920e-02
            -9.386945838351230531e-01 3.357676008295389924e-01 1.819932192611799920e-02
            9.386945838351230531e-01 -3.357676008295389924e-01 1.819932192611799920e-02
            -9.386945838351230531e-01 -3.357676008295389924e-01 1.819932192611799920e-02
            3.357676008295389924e-01 9.386945838351230531e-01 1.819932192611799920e-02
            -3.357676008295389924e-01 9.386945838351230531e-01 1.819932192611799920e-02
            3.357676008295389924e-01 -9.386945838351230531e-01 1.819932192611799920e-02
            -3.357676008295389924e-01 -9.386945838351230531e-01 1.819932192611799920e-02
            7.520427768039540384e-01 3.797170111700770012e-01 9.578070593943299738e-02
            -7.520427768039540384e-01 3.797170111700770012e-01 9.578070593943299738e-02
            7.520427768039540384e-01 -3.797170111700770012e-01 9.578070593943299738e-02
            -7.520427768039540384e-01 -3.797170111700770012e-01 9.578070593943299738e-02
            3.797170111700770012e-01 7.520427768039540384e-01 9.578070593943299738e-02
            -3.797170111700770012e-01 7.520427768039540384e-01 9.578070593943299738e-02
            3.797170111700770012e-01 -7.520427768039540384e-01 9.578070593943299738e-02
            -3.797170111700770012e-01 -7.520427768039540384e-01 9.578070593943299738e-02
            ];

    case  13

        % moment matching: 1.8e-13
        % cardinality:  36
        % max norm point: 9.97e-01

        XYW=[
            2.834028323488250178e-01 0.000000000000000000e+00 1.562398698373330119e-01
            -2.834028323488250178e-01 0.000000000000000000e+00 1.562398698373330119e-01
            0.000000000000000000e+00 2.834028323488250178e-01 1.562398698373330119e-01
            0.000000000000000000e+00 -2.834028323488250178e-01 1.562398698373330119e-01
            9.205754740362509514e-01 0.000000000000000000e+00 6.944740926697699768e-02
            -9.205754740362509514e-01 0.000000000000000000e+00 6.944740926697699768e-02
            0.000000000000000000e+00 9.205754740362509514e-01 6.944740926697699768e-02
            0.000000000000000000e+00 -9.205754740362509514e-01 6.944740926697699768e-02
            6.490072305780020256e-01 0.000000000000000000e+00 1.381459105348709915e-01
            -6.490072305780020256e-01 0.000000000000000000e+00 1.381459105348709915e-01
            0.000000000000000000e+00 6.490072305780020256e-01 1.381459105348709915e-01
            0.000000000000000000e+00 -6.490072305780020256e-01 1.381459105348709915e-01
            6.773551060280690317e-01 6.773551060280690317e-01 4.337658388715499858e-02
            -6.773551060280690317e-01 6.773551060280690317e-01 4.337658388715499858e-02
            6.773551060280690317e-01 -6.773551060280690317e-01 4.337658388715499858e-02
            -6.773551060280690317e-01 -6.773551060280690317e-01 4.337658388715499858e-02
            4.126722167632890237e-01 4.126722167632890237e-01 1.502283341400099947e-01
            -4.126722167632890237e-01 4.126722167632890237e-01 1.502283341400099947e-01
            4.126722167632890237e-01 -4.126722167632890237e-01 1.502283341400099947e-01
            -4.126722167632890237e-01 -4.126722167632890237e-01 1.502283341400099947e-01
            9.386945838351230531e-01 3.357676008295389924e-01 1.819932192611799920e-02
            -9.386945838351230531e-01 3.357676008295389924e-01 1.819932192611799920e-02
            9.386945838351230531e-01 -3.357676008295389924e-01 1.819932192611799920e-02
            -9.386945838351230531e-01 -3.357676008295389924e-01 1.819932192611799920e-02
            3.357676008295389924e-01 9.386945838351230531e-01 1.819932192611799920e-02
            -3.357676008295389924e-01 9.386945838351230531e-01 1.819932192611799920e-02
            3.357676008295389924e-01 -9.386945838351230531e-01 1.819932192611799920e-02
            -3.357676008295389924e-01 -9.386945838351230531e-01 1.819932192611799920e-02
            7.520427768039540384e-01 3.797170111700770012e-01 9.578070593943299738e-02
            -7.520427768039540384e-01 3.797170111700770012e-01 9.578070593943299738e-02
            7.520427768039540384e-01 -3.797170111700770012e-01 9.578070593943299738e-02
            -7.520427768039540384e-01 -3.797170111700770012e-01 9.578070593943299738e-02
            3.797170111700770012e-01 7.520427768039540384e-01 9.578070593943299738e-02
            -3.797170111700770012e-01 7.520427768039540384e-01 9.578070593943299738e-02
            3.797170111700770012e-01 -7.520427768039540384e-01 9.578070593943299738e-02
            -3.797170111700770012e-01 -7.520427768039540384e-01 9.578070593943299738e-02
            ];

    case  14

        % moment matching: 1.1e-12
        % cardinality:  44
        % max norm point: 9.90e-01

        XYW=[
            2.528637970912929744e-01 0.000000000000000000e+00 1.252902085643380070e-01
            -2.528637970912929744e-01 0.000000000000000000e+00 1.252902085643380070e-01
            0.000000000000000000e+00 2.528637970912929744e-01 1.252902085643380070e-01
            0.000000000000000000e+00 -2.528637970912929744e-01 1.252902085643380070e-01
            9.897468025116139545e-01 0.000000000000000000e+00 1.671262549698199901e-02
            -9.897468025116139545e-01 0.000000000000000000e+00 1.671262549698199901e-02
            0.000000000000000000e+00 9.897468025116139545e-01 1.671262549698199901e-02
            0.000000000000000000e+00 -9.897468025116139545e-01 1.671262549698199901e-02
            5.777289284449580453e-01 0.000000000000000000e+00 1.095003911263650043e-01
            -5.777289284449580453e-01 0.000000000000000000e+00 1.095003911263650043e-01
            0.000000000000000000e+00 5.777289284449580453e-01 1.095003911263650043e-01
            0.000000000000000000e+00 -5.777289284449580453e-01 1.095003911263650043e-01
            8.738369566450350234e-01 0.000000000000000000e+00 6.623745579639700454e-02
            -8.738369566450350234e-01 0.000000000000000000e+00 6.623745579639700454e-02
            0.000000000000000000e+00 8.738369566450350234e-01 6.623745579639700454e-02
            0.000000000000000000e+00 -8.738369566450350234e-01 6.623745579639700454e-02
            6.892993807911359516e-01 6.892993807911359516e-01 2.610286018435799965e-02
            -6.892993807911359516e-01 6.892993807911359516e-01 2.610286018435799965e-02
            6.892993807911359516e-01 -6.892993807911359516e-01 2.610286018435799965e-02
            -6.892993807911359516e-01 -6.892993807911359516e-01 2.610286018435799965e-02
            5.976143046672079961e-01 5.976143046672079961e-01 6.600093466110000529e-02
            -5.976143046672079961e-01 5.976143046672079961e-01 6.600093466110000529e-02
            5.976143046672079961e-01 -5.976143046672079961e-01 6.600093466110000529e-02
            -5.976143046672079961e-01 -5.976143046672079961e-01 6.600093466110000529e-02
            3.754168246261699871e-01 3.754168246261699871e-01 1.274283726817199980e-01
            -3.754168246261699871e-01 3.754168246261699871e-01 1.274283726817199980e-01
            3.754168246261699871e-01 -3.754168246261699871e-01 1.274283726817199980e-01
            -3.754168246261699871e-01 -3.754168246261699871e-01 1.274283726817199980e-01
            8.830971113185910326e-01 3.657908004006629743e-01 4.252306582668100021e-02
            -8.830971113185910326e-01 3.657908004006629743e-01 4.252306582668100021e-02
            8.830971113185910326e-01 -3.657908004006629743e-01 4.252306582668100021e-02
            -8.830971113185910326e-01 -3.657908004006629743e-01 4.252306582668100021e-02
            3.657908004006629743e-01 8.830971113185910326e-01 4.252306582668100021e-02
            -3.657908004006629743e-01 8.830971113185910326e-01 4.252306582668100021e-02
            3.657908004006629743e-01 -8.830971113185910326e-01 4.252306582668100021e-02
            -3.657908004006629743e-01 -8.830971113185910326e-01 4.252306582668100021e-02
            7.074387449600699762e-01 2.930307227106639933e-01 8.153959161641299858e-02
            -7.074387449600699762e-01 2.930307227106639933e-01 8.153959161641299858e-02
            7.074387449600699762e-01 -2.930307227106639933e-01 8.153959161641299858e-02
            -7.074387449600699762e-01 -2.930307227106639933e-01 8.153959161641299858e-02
            2.930307227106639933e-01 7.074387449600699762e-01 8.153959161641299858e-02
            -2.930307227106639933e-01 7.074387449600699762e-01 8.153959161641299858e-02
            2.930307227106639933e-01 -7.074387449600699762e-01 8.153959161641299858e-02
            -2.930307227106639933e-01 -7.074387449600699762e-01 8.153959161641299858e-02
            ];

    case  15

        % moment matching: 1.1e-12
        % cardinality:  44
        % max norm point: 9.90e-01

        XYW=[
            2.528637970912929744e-01 0.000000000000000000e+00 1.252902085643380070e-01
            -2.528637970912929744e-01 0.000000000000000000e+00 1.252902085643380070e-01
            0.000000000000000000e+00 2.528637970912929744e-01 1.252902085643380070e-01
            0.000000000000000000e+00 -2.528637970912929744e-01 1.252902085643380070e-01
            9.897468025116139545e-01 0.000000000000000000e+00 1.671262549698199901e-02
            -9.897468025116139545e-01 0.000000000000000000e+00 1.671262549698199901e-02
            0.000000000000000000e+00 9.897468025116139545e-01 1.671262549698199901e-02
            0.000000000000000000e+00 -9.897468025116139545e-01 1.671262549698199901e-02
            5.777289284449580453e-01 0.000000000000000000e+00 1.095003911263650043e-01
            -5.777289284449580453e-01 0.000000000000000000e+00 1.095003911263650043e-01
            0.000000000000000000e+00 5.777289284449580453e-01 1.095003911263650043e-01
            0.000000000000000000e+00 -5.777289284449580453e-01 1.095003911263650043e-01
            8.738369566450350234e-01 0.000000000000000000e+00 6.623745579639700454e-02
            -8.738369566450350234e-01 0.000000000000000000e+00 6.623745579639700454e-02
            0.000000000000000000e+00 8.738369566450350234e-01 6.623745579639700454e-02
            0.000000000000000000e+00 -8.738369566450350234e-01 6.623745579639700454e-02
            6.892993807911359516e-01 6.892993807911359516e-01 2.610286018435799965e-02
            -6.892993807911359516e-01 6.892993807911359516e-01 2.610286018435799965e-02
            6.892993807911359516e-01 -6.892993807911359516e-01 2.610286018435799965e-02
            -6.892993807911359516e-01 -6.892993807911359516e-01 2.610286018435799965e-02
            5.976143046672079961e-01 5.976143046672079961e-01 6.600093466110000529e-02
            -5.976143046672079961e-01 5.976143046672079961e-01 6.600093466110000529e-02
            5.976143046672079961e-01 -5.976143046672079961e-01 6.600093466110000529e-02
            -5.976143046672079961e-01 -5.976143046672079961e-01 6.600093466110000529e-02
            3.754168246261699871e-01 3.754168246261699871e-01 1.274283726817199980e-01
            -3.754168246261699871e-01 3.754168246261699871e-01 1.274283726817199980e-01
            3.754168246261699871e-01 -3.754168246261699871e-01 1.274283726817199980e-01
            -3.754168246261699871e-01 -3.754168246261699871e-01 1.274283726817199980e-01
            8.830971113185910326e-01 3.657908004006629743e-01 4.252306582668100021e-02
            -8.830971113185910326e-01 3.657908004006629743e-01 4.252306582668100021e-02
            8.830971113185910326e-01 -3.657908004006629743e-01 4.252306582668100021e-02
            -8.830971113185910326e-01 -3.657908004006629743e-01 4.252306582668100021e-02
            3.657908004006629743e-01 8.830971113185910326e-01 4.252306582668100021e-02
            -3.657908004006629743e-01 8.830971113185910326e-01 4.252306582668100021e-02
            3.657908004006629743e-01 -8.830971113185910326e-01 4.252306582668100021e-02
            -3.657908004006629743e-01 -8.830971113185910326e-01 4.252306582668100021e-02
            7.074387449600699762e-01 2.930307227106639933e-01 8.153959161641299858e-02
            -7.074387449600699762e-01 2.930307227106639933e-01 8.153959161641299858e-02
            7.074387449600699762e-01 -2.930307227106639933e-01 8.153959161641299858e-02
            -7.074387449600699762e-01 -2.930307227106639933e-01 8.153959161641299858e-02
            2.930307227106639933e-01 7.074387449600699762e-01 8.153959161641299858e-02
            -2.930307227106639933e-01 7.074387449600699762e-01 8.153959161641299858e-02
            2.930307227106639933e-01 -7.074387449600699762e-01 8.153959161641299858e-02
            -2.930307227106639933e-01 -7.074387449600699762e-01 8.153959161641299858e-02
            ];



end









function XYW=cub_rules_stroud_1971(deg)

%--------------------------------------------------------------------------
% Reference
% A.H. Stroud.
% Approximate Calculation of Multiple Integrals.
% Prentice Hall, Englewood Cliffs, NH, 1971.
%--------------------------------------------------------------------------
% | Deg | Card | type|
% .....................
% |  2  |  3  |  PI  |
% |  3  |  4  |  PI  |
% |  4  |  6  |  PI  |
% |  5  |  7  |  PI  |
% |  7  | 12  |  PI  |
%--------------------------------------------------------------------------



switch deg

    case 2

        % moment matching: 2.8e-15
        % cardinality:   3
        % max norm point: 7.07e-01

        XYW=[
            0.500000000000000  0.500000000000000  1.047197551196598    0.70711
            -0.683012701892219  0.183012701892219  1.047197551196598    0.70711
            0.183012701892219 -0.683012701892219  1.047197551196598    0.70711
            ];

    case 3

        % moment matching: 2.7e-15
        % cardinality:   4
        % max norm point: 7.07e-01

        XYW=[
            0.707106781186547 -0.000000000000000  0.785398163397448    0.70711  1.00000
            -0.707106781186547 -0.000000000000000  0.785398163397448    0.70711 -1.00000
            -0.000000000000000  0.707106781186547  0.785398163397448    0.70711  0.00000
            -0.000000000000000 -0.707106781186547  0.785398163397448    0.70711  0.00000
            ];

    case 4

        % moment matching: 6.7e-16
        % cardinality:   6
        % max norm point: 8.16e-01

        XYW=[
            -0.000000000000000 -0.000000000000000  0.785398163397448    0.00000  0.00000
            -0.816496580927726 -0.000000000000000  0.471238898038469    0.81650  1.00000
            -0.252311319355707 -0.776534393824027  0.471238898038469    0.81650  0.30902
            -0.252311319355707  0.776534393824027  0.471238898038469    0.81650  0.30902
            0.660559609819570  0.479924648816545  0.471238898038469    0.81650 -0.80902
            0.660559609819570 -0.479924648816545  0.471238898038469    0.81650 -0.80902
            ];

    case 5

        % moment matching: 2.4e-15
        % cardinality:   7
        % max norm point: 8.16e-01

        XYW=[
            0.000000000000000  0.000000000000000  0.785398163397448    0.00000  0.00000
            -0.707106781186548  0.408248290463863  0.392699081698724    0.81650  1.00000
            0.000000000000000 -0.816496580927726  0.392699081698724    0.81650 -0.50000
            0.000000000000000  0.816496580927726  0.392699081698724    0.81650  0.50000
            0.707106781186548 -0.408248290463863  0.392699081698724    0.81650 -1.00000
            0.707106781186548  0.408248290463863  0.392699081698724    0.81650 -0.50000
            -0.707106781186548 -0.408248290463863  0.392699081698724    0.81650  0.50000
            ];

    case 7

        % moment matching: 5.9e-15
        % cardinality:  12
        % max norm point: 9.11e-01

        XYW=[
            0.866025403784438  0.000000000000000  0.232710566932577    0.86603  1.00000  0.70711  0.70711
            -0.866025403784438  0.000000000000000  0.232710566932577    0.86603 -1.00000 -0.70711 -0.70711
            0.000000000000000  0.866025403784439  0.232710566932577    0.86603  0.00000  0.70711  0.70711
            0.000000000000000 -0.866025403784439  0.232710566932577    0.86603  0.00000 -0.70711 -0.70711
            0.322914992067413  0.322914992067413  0.387077796006217    0.45667  0.70711  1.00000  1.00000
            -0.322914992067388  0.322914992067389  0.387077796006236    0.45667 -0.70711  0.00000  0.00000
            0.322914992067388 -0.322914992067389  0.387077796006236    0.45667  0.70711 -0.00000 -0.00000
            -0.322914992067413 -0.322914992067414  0.387077796006217    0.45667 -0.70711 -1.00000 -1.00000
            0.644171310389468  0.644171310389468  0.165609800458638    0.91100  0.70711  1.00000  1.00000
            -0.644171310389461  0.644171310389461  0.165609800458651    0.91100 -0.70711  0.00000 -0.00000
            0.644171310389461 -0.644171310389461  0.165609800458651    0.91100  0.70711 -0.00000  0.00000
            -0.644171310389469 -0.644171310389468  0.165609800458637    0.91100 -0.70711 -1.00000 -1.00000
            ];

end

XYW=XYW(:,1:3);








function XYW=cub_rules_piessens_haegemans(deg)

%--------------------------------------------------------------------------
% Reference
% Construction of cubature formulas of degree eleven for symmetric planar
% regions, using orthogonal polynomials
% Ann Haegemans & Robert Piessens
% Numerische Mathematik volume 25, pages139?148 (1976)
%--------------------------------------------------------------------------
% | Deg | Card | type|
% .....................
% |  11 |  26 |  PI* |
%--------------------------------------------------------------------------

switch deg

    case 11

        % moment matching: 4.1e-10
        % cardinality:  26
        % max norm point: 9.77e-01

        % XYW0=[7.700399234000000e-01     2.077544664000000e-01     1.124135813000000e-01
        %      4.332553445000000e-01     4.967812243000000e-01     1.741604140000000e-01
        %      8.042027006000000e-01     5.047957527000000e-01     6.356382332000000e-02
        %      4.487567036000000e-01     8.138343334000000e-01     8.475275483000000e-02
        %      9.774742944000000e-01                         0     4.192925977000000e-02
        %      4.222292895000000e-01                         0     2.108288631000000e-01
        %                          0     2.757676283000000e-01     2.337746724000000e-01
        %                          0     7.248280902999999e-01     1.582130408000000e-01
        %                          0     9.564095370000000e-01     5.626934370000000e-01];


        XYW=[
            -7.700399234000000e-01    -2.077544664000000e-01     1.124135813000000e-01
            -7.700399234000000e-01     2.077544664000000e-01     1.124135813000000e-01
            7.700399234000000e-01    -2.077544664000000e-01     1.124135813000000e-01
            7.700399234000000e-01     2.077544664000000e-01     1.124135813000000e-01
            -4.332553445000000e-01    -4.967812243000000e-01     1.741604140000000e-01
            -4.332553445000000e-01     4.967812243000000e-01     1.741604140000000e-01
            4.332553445000000e-01    -4.967812243000000e-01     1.741604140000000e-01
            4.332553445000000e-01     4.967812243000000e-01     1.741604140000000e-01
            -8.042027006000000e-01    -5.047957527000000e-01     6.356382332000000e-02
            -8.042027006000000e-01     5.047957527000000e-01     6.356382332000000e-02
            8.042027006000000e-01    -5.047957527000000e-01     6.356382332000000e-02
            8.042027006000000e-01     5.047957527000000e-01     6.356382332000000e-02
            -4.487567036000000e-01    -8.138343334000000e-01     8.475275483000000e-02
            -4.487567036000000e-01     8.138343334000000e-01     8.475275483000000e-02
            4.487567036000000e-01    -8.138343334000000e-01     8.475275483000000e-02
            4.487567036000000e-01     8.138343334000000e-01     8.475275483000000e-02
            -9.774742944000000e-01                         0     4.192925977000000e-02
            9.774742944000000e-01                         0     4.192925977000000e-02
            -4.222292895000000e-01                         0     2.108288631000000e-01
            4.222292895000000e-01                         0     2.108288631000000e-01
            0    -2.757676283000000e-01     2.337746724000000e-01
            0     2.757676283000000e-01     2.337746724000000e-01
            0    -7.248280902999999e-01     1.582130408000000e-01
            0     7.248280902999999e-01     1.582130408000000e-01
            0    -9.564095370000000e-01     5.626934370000000e-02
            0     9.564095370000000e-01     5.626934370000000e-02];


end










function XYW=cub_rules_nschloe(deg)

% .........................................................................
% Object:
% Low cardinality cubature sets for cubature on the unit-disk, with respect
% to the Legendre measure "w(x,y)=1", taken from:
%          https://awesomeopensource.com/project/nschloe/quadpy
%
% These sets were converted in Matlab from quadpy.
% .........................................................................

% ...................................
% | Deg | Card | type |
% ...................................
% |  1 |    1 | 0.0e+00 | 0.00e+00 |
% | 13 |   36 | 1.7e-15 | 9.97e-01 |
% | 18 |   72 | 1.6e-15 | 9.92e-01 |
% | 19 |   72 | 1.7e-15 | 9.92e-01 |
% ...................................

switch deg

    case 1

        XYW=[0 0 pi];

    case 6

        XYW=[     8.660254037844390e-01                         0     2.327105669325773e-01
            -8.660254037844390e-01                         0     2.327105669325773e-01
            0     8.660254037844390e-01     2.327105669325773e-01
            0    -8.660254037844390e-01     2.327105669325773e-01
            3.229149920674000e-01     3.229149920674000e-01     3.870777960062258e-01
            6.441713103894650e-01     6.441713103894650e-01     1.656098004586450e-01
            -3.229149920674000e-01     3.229149920674000e-01     3.870777960062258e-01
            -6.441713103894650e-01     6.441713103894650e-01     1.656098004586450e-01
            3.229149920674000e-01    -3.229149920674000e-01     3.870777960062258e-01
            6.441713103894650e-01    -6.441713103894650e-01     1.656098004586450e-01
            -3.229149920674000e-01    -3.229149920674000e-01     3.870777960062258e-01
            -6.441713103894650e-01    -6.441713103894650e-01     1.656098004586450e-01];

    case 13

        XYW=[ 3.357676008295438e-01     9.386945838351065e-01     5.793024090926356e-03
            3.797170111700796e-01     7.520427768039379e-01     3.048794560618420e-02
            -3.357676008295438e-01     9.386945838351065e-01     5.793024090926356e-03
            -3.797170111700796e-01     7.520427768039379e-01     3.048794560618420e-02
            3.357676008295438e-01    -9.386945838351065e-01     5.793024090926356e-03
            3.797170111700796e-01    -7.520427768039379e-01     3.048794560618420e-02
            -3.357676008295438e-01    -9.386945838351065e-01     5.793024090926356e-03
            -3.797170111700796e-01    -7.520427768039379e-01     3.048794560618420e-02
            9.386945838351065e-01     3.357676008295438e-01     5.793024090926356e-03
            7.520427768039379e-01     3.797170111700796e-01     3.048794560618420e-02
            -9.386945838351065e-01     3.357676008295438e-01     5.793024090926356e-03
            -7.520427768039379e-01     3.797170111700796e-01     3.048794560618420e-02
            9.386945838351065e-01    -3.357676008295438e-01     5.793024090926356e-03
            7.520427768039379e-01    -3.797170111700796e-01     3.048794560618420e-02
            -9.386945838351065e-01    -3.357676008295438e-01     5.793024090926356e-03
            -7.520427768039379e-01    -3.797170111700796e-01     3.048794560618420e-02
            6.773551060280696e-01     6.773551060280696e-01     1.380719548016184e-02
            4.126722167632789e-01     4.126722167632789e-01     4.781916394168508e-02
            -6.773551060280696e-01     6.773551060280696e-01     1.380719548016184e-02
            -4.126722167632789e-01     4.126722167632789e-01     4.781916394168508e-02
            6.773551060280696e-01    -6.773551060280696e-01     1.380719548016184e-02
            4.126722167632789e-01    -4.126722167632789e-01     4.781916394168508e-02
            -6.773551060280696e-01    -6.773551060280696e-01     1.380719548016184e-02
            -4.126722167632789e-01    -4.126722167632789e-01     4.781916394168508e-02
            2.834028323488404e-01                         0     4.973269518529444e-02
            6.490072305780239e-01                         0     4.397320905910847e-02
            9.205754740362561e-01                         0     2.210579693952910e-02
            -2.834028323488404e-01                         0     4.973269518529444e-02
            -6.490072305780239e-01                         0     4.397320905910847e-02
            -9.205754740362561e-01                         0     2.210579693952910e-02
            0     2.834028323488404e-01     4.973269518529444e-02
            0     6.490072305780239e-01     4.397320905910847e-02
            0     9.205754740362561e-01     2.210579693952910e-02
            0    -2.834028323488404e-01     4.973269518529444e-02
            0    -6.490072305780239e-01     4.397320905910847e-02
            0    -9.205754740362561e-01     2.210579693952910e-02];

        XYW(:,3)=pi*XYW(:,3);


    case 18

        XYW=[     2.046700480083686e-01                         0     2.627953045897823e-02
            9.923095462982181e-01                         0     3.094532283805032e-03
            7.409306345221045e-01                         0     1.971001826894905e-02
            4.779883150369948e-01                         0     2.518556913389217e-02
            -2.046700480083686e-01                         0     2.627953045897823e-02
            -9.923095462982181e-01                         0     3.094532283805032e-03
            -7.409306345221045e-01                         0     1.971001826894905e-02
            -4.779883150369948e-01                         0     2.518556913389217e-02
            0     2.046700480083686e-01     2.627953045897823e-02
            0     9.923095462982181e-01     3.094532283805032e-03
            0     7.409306345221045e-01     1.971001826894905e-02
            0     4.779883150369948e-01     2.518556913389217e-02
            0    -2.046700480083686e-01     2.627953045897823e-02
            0    -9.923095462982181e-01     3.094532283805032e-03
            0    -7.409306345221045e-01     1.971001826894905e-02
            0    -4.779883150369948e-01     2.518556913389217e-02
            3.061394918787155e-01     3.061394918787155e-01     2.786051397558758e-02
            5.247826250978556e-01     5.247826250978556e-01     1.816835336726267e-02
            -3.061394918787155e-01     3.061394918787155e-01     2.786051397558758e-02
            -5.247826250978556e-01     5.247826250978556e-01     1.816835336726267e-02
            3.061394918787155e-01    -3.061394918787155e-01     2.786051397558758e-02
            5.247826250978556e-01    -5.247826250978556e-01     1.816835336726267e-02
            -3.061394918787155e-01    -3.061394918787155e-01     2.786051397558758e-02
            -5.247826250978556e-01    -5.247826250978556e-01     1.816835336726267e-02
            9.218059338256883e-01     3.109205531727743e-01     6.678727854795258e-03
            7.902364837078591e-01     5.798985587983276e-01     4.846472990348523e-03
            7.257934979678180e-01     5.250477025639917e-01     1.054767394843740e-02
            7.882284597939131e-01     2.902466240685259e-01     1.427750044099094e-02
            5.848934178476086e-01     2.643178362927081e-01     2.079243557976361e-02
            9.096365099864627e-01     9.257372194320643e-02     7.707930441426864e-03
            -9.218059338256883e-01     3.109205531727743e-01     6.678727854795258e-03
            -7.902364837078591e-01     5.798985587983276e-01     4.846472990348523e-03
            -7.257934979678180e-01     5.250477025639917e-01     1.054767394843740e-02
            -7.882284597939131e-01     2.902466240685259e-01     1.427750044099094e-02
            -5.848934178476086e-01     2.643178362927081e-01     2.079243557976361e-02
            -9.096365099864627e-01     9.257372194320643e-02     7.707930441426864e-03
            9.218059338256883e-01    -3.109205531727743e-01     6.678727854795258e-03
            7.902364837078591e-01    -5.798985587983276e-01     4.846472990348523e-03
            7.257934979678180e-01    -5.250477025639917e-01     1.054767394843740e-02
            7.882284597939131e-01    -2.902466240685259e-01     1.427750044099094e-02
            5.848934178476086e-01    -2.643178362927081e-01     2.079243557976361e-02
            9.096365099864627e-01    -9.257372194320643e-02     7.707930441426864e-03
            -9.218059338256883e-01    -3.109205531727743e-01     6.678727854795258e-03
            -7.902364837078591e-01    -5.798985587983276e-01     4.846472990348523e-03
            -7.257934979678180e-01    -5.250477025639917e-01     1.054767394843740e-02
            -7.882284597939131e-01    -2.902466240685259e-01     1.427750044099094e-02
            -5.848934178476086e-01    -2.643178362927081e-01     2.079243557976361e-02
            -9.096365099864627e-01    -9.257372194320643e-02     7.707930441426864e-03
            3.109205531727743e-01     9.218059338256883e-01     6.678727854795258e-03
            5.798985587983276e-01     7.902364837078591e-01     4.846472990348523e-03
            5.250477025639917e-01     7.257934979678180e-01     1.054767394843740e-02
            2.902466240685259e-01     7.882284597939131e-01     1.427750044099094e-02
            2.643178362927081e-01     5.848934178476086e-01     2.079243557976361e-02
            9.257372194320643e-02     9.096365099864627e-01     7.707930441426864e-03
            -3.109205531727743e-01     9.218059338256883e-01     6.678727854795258e-03
            -5.798985587983276e-01     7.902364837078591e-01     4.846472990348523e-03
            -5.250477025639917e-01     7.257934979678180e-01     1.054767394843740e-02
            -2.902466240685259e-01     7.882284597939131e-01     1.427750044099094e-02
            -2.643178362927081e-01     5.848934178476086e-01     2.079243557976361e-02
            -9.257372194320643e-02     9.096365099864627e-01     7.707930441426864e-03
            3.109205531727743e-01    -9.218059338256883e-01     6.678727854795258e-03
            5.798985587983276e-01    -7.902364837078591e-01     4.846472990348523e-03
            5.250477025639917e-01    -7.257934979678180e-01     1.054767394843740e-02
            2.902466240685259e-01    -7.882284597939131e-01     1.427750044099094e-02
            2.643178362927081e-01    -5.848934178476086e-01     2.079243557976361e-02
            9.257372194320643e-02    -9.096365099864627e-01     7.707930441426864e-03
            -3.109205531727743e-01    -9.218059338256883e-01     6.678727854795258e-03
            -5.798985587983276e-01    -7.902364837078591e-01     4.846472990348523e-03
            -5.250477025639917e-01    -7.257934979678180e-01     1.054767394843740e-02
            -2.902466240685259e-01    -7.882284597939131e-01     1.427750044099094e-02
            -2.643178362927081e-01    -5.848934178476086e-01     2.079243557976361e-02
            -9.257372194320643e-02    -9.096365099864627e-01     7.707930441426864e-03];

        XYW(:,3)=pi*XYW(:,3);

    case 19

        XYW=[     2.046700480083686e-01                         0     8.255957982971522e-02
            9.923095462982181e-01                         0     9.721759889098333e-03
            7.409306345221045e-01                         0     6.192084859585094e-02
            4.779883150369948e-01                         0     7.912279896751348e-02
            -2.046700480083686e-01                         0     8.255957982971522e-02
            -9.923095462982181e-01                         0     9.721759889098333e-03
            -7.409306345221045e-01                         0     6.192084859585094e-02
            -4.779883150369948e-01                         0     7.912279896751348e-02
            0     2.046700480083686e-01     8.255957982971522e-02
            0     9.923095462982181e-01     9.721759889098333e-03
            0     7.409306345221045e-01     6.192084859585094e-02
            0     4.779883150369948e-01     7.912279896751348e-02
            0    -2.046700480083686e-01     8.255957982971522e-02
            0    -9.923095462982181e-01     9.721759889098333e-03
            0    -7.409306345221045e-01     6.192084859585094e-02
            0    -4.779883150369948e-01     7.912279896751348e-02
            3.061394918787155e-01     3.061394918787155e-01     8.752638603094169e-02
            5.247826250978556e-01     5.247826250978556e-01     5.707756546641578e-02
            -3.061394918787155e-01     3.061394918787155e-01     8.752638603094169e-02
            -5.247826250978556e-01     5.247826250978556e-01     5.707756546641578e-02
            3.061394918787155e-01    -3.061394918787155e-01     8.752638603094169e-02
            5.247826250978556e-01    -5.247826250978556e-01     5.707756546641578e-02
            -3.061394918787155e-01    -3.061394918787155e-01     8.752638603094169e-02
            -5.247826250978556e-01    -5.247826250978556e-01     5.707756546641578e-02
            9.218059338256883e-01     3.109205531727743e-01     2.098184236395030e-02
            7.902364837078591e-01     5.798985587983276e-01     1.522564394230028e-02
            7.257934979678180e-01     5.250477025639917e-01     3.313649498887137e-02
            7.882284597939131e-01     2.902466240685259e-01     4.485409049704218e-02
            5.848934178476086e-01     2.643178362927081e-01     6.532136286762440e-02
            9.096365099864627e-01     9.257372194320643e-02     2.421517764916777e-02
            -9.218059338256883e-01     3.109205531727743e-01     2.098184236395030e-02
            -7.902364837078591e-01     5.798985587983276e-01     1.522564394230028e-02
            -7.257934979678180e-01     5.250477025639917e-01     3.313649498887137e-02
            -7.882284597939131e-01     2.902466240685259e-01     4.485409049704218e-02
            -5.848934178476086e-01     2.643178362927081e-01     6.532136286762440e-02
            -9.096365099864627e-01     9.257372194320643e-02     2.421517764916777e-02
            9.218059338256883e-01    -3.109205531727743e-01     2.098184236395030e-02
            7.902364837078591e-01    -5.798985587983276e-01     1.522564394230028e-02
            7.257934979678180e-01    -5.250477025639917e-01     3.313649498887137e-02
            7.882284597939131e-01    -2.902466240685259e-01     4.485409049704218e-02
            5.848934178476086e-01    -2.643178362927081e-01     6.532136286762440e-02
            9.096365099864627e-01    -9.257372194320643e-02     2.421517764916777e-02
            -9.218059338256883e-01    -3.109205531727743e-01     2.098184236395030e-02
            -7.902364837078591e-01    -5.798985587983276e-01     1.522564394230028e-02
            -7.257934979678180e-01    -5.250477025639917e-01     3.313649498887137e-02
            -7.882284597939131e-01    -2.902466240685259e-01     4.485409049704218e-02
            -5.848934178476086e-01    -2.643178362927081e-01     6.532136286762440e-02
            -9.096365099864627e-01    -9.257372194320643e-02     2.421517764916777e-02
            3.109205531727743e-01     9.218059338256883e-01     2.098184236395030e-02
            5.798985587983276e-01     7.902364837078591e-01     1.522564394230028e-02
            5.250477025639917e-01     7.257934979678180e-01     3.313649498887137e-02
            2.902466240685259e-01     7.882284597939131e-01     4.485409049704218e-02
            2.643178362927081e-01     5.848934178476086e-01     6.532136286762440e-02
            9.257372194320643e-02     9.096365099864627e-01     2.421517764916777e-02
            -3.109205531727743e-01     9.218059338256883e-01     2.098184236395030e-02
            -5.798985587983276e-01     7.902364837078591e-01     1.522564394230028e-02
            -5.250477025639917e-01     7.257934979678180e-01     3.313649498887137e-02
            -2.902466240685259e-01     7.882284597939131e-01     4.485409049704218e-02
            -2.643178362927081e-01     5.848934178476086e-01     6.532136286762440e-02
            -9.257372194320643e-02     9.096365099864627e-01     2.421517764916777e-02
            3.109205531727743e-01    -9.218059338256883e-01     2.098184236395030e-02
            5.798985587983276e-01    -7.902364837078591e-01     1.522564394230028e-02
            5.250477025639917e-01    -7.257934979678180e-01     3.313649498887137e-02
            2.902466240685259e-01    -7.882284597939131e-01     4.485409049704218e-02
            2.643178362927081e-01    -5.848934178476086e-01     6.532136286762440e-02
            9.257372194320643e-02    -9.096365099864627e-01     2.421517764916777e-02
            -3.109205531727743e-01    -9.218059338256883e-01     2.098184236395030e-02
            -5.798985587983276e-01    -7.902364837078591e-01     1.522564394230028e-02
            -5.250477025639917e-01    -7.257934979678180e-01     3.313649498887137e-02
            -2.902466240685259e-01    -7.882284597939131e-01     4.485409049704218e-02
            -2.643178362927081e-01    -5.848934178476086e-01     6.532136286762440e-02
            -9.257372194320643e-02    -9.096365099864627e-01     2.421517764916777e-02];


end




function xyw=unit_disk_rules_takaki_forbes_rolland(deg)

%--------------------------------------------------------------------------
% Reference paper:
% Schemes for cubature over the unit disk found via numerical optimization
% N. Takaki, G.W. Forbes, Jannick P. Rolland
% Volume 407, June 2022, 114076
% Journal of Computational and Applied Mathematics
%--------------------------------------------------------------------------

%------------------------------------------
%  deg   card    dim     err     larg.rad.
%------------------------------------------
% | 17 |   55 |  171 | 5.4e-15 | 9.83e-01 |
% | 21 |   85 |  253 | 1.0e-14 | 9.90e-01 |
% | 25 |  117 |  351 | 3.2e-13 | 9.95e-01 |
% | 27 |  137 |  406 | 1.5e-13 | 9.99e-01 |
% | 29 |  155 |  465 | 2.9e-13 | 9.95e-01 |
% | 33 |  199 |  595 | 1.8e-13 | 9.96e-01 |
% | 37 |  247 |  741 | 1.7e-14 | 9.97e-01 |
% | 39 |  273 |  820 | 1.5e-13 | 9.99e-01 |
% | 41 |  295 |  903 | 1.8e-14 | 9.97e-01 |
% | 45 |  361 | 1081 | 1.8e-13 | 9.98e-01 |
% | 49 |  425 | 1275 | 1.9e-13 | 9.99e-01 |
% | 53 |  487 | 1485 | 2.6e-14 | 9.99e-01 |
% | 65 |  733 | 2211 | 2.9e-14 | 9.99e-01 |
% | 77 | 1021 | 3081 | 3.4e-14 | 9.99e-01 |
%------------------------------------------

degV=[17 21 25 27 29 33 37 39 41 45 49 53 65 77];

in=ismember(deg,degV);

if in == 0
    fprintf('\n \t Degree not in the set');
    xyw=[];
end


switch deg

    case 17
        xyw=[                 0                         0     1.199138901137160e-01
            3.637754205500280e-01                         0     1.093903749000560e-01
            5.227928566630328e-01     2.979667740679184e-01     8.960586792212499e-02
            6.780470283789560e-01    -7.402882492014267e-03     8.090154301058200e-02
            7.761865069877990e-01     2.880757874447525e-01     5.466292404036400e-02
            6.305235782117016e-01     5.585972967012899e-01     5.634122793587900e-02
            8.926437817839168e-01     3.439072732955102e-02     4.252012057136800e-02
            8.777647679474704e-01     4.024520628881447e-01     2.931195327772600e-02
            6.918014423063982e-01     6.829326253808480e-01     2.467100183788900e-02
            9.788244781051294e-01     8.799120039872618e-02     1.620811375002300e-02
            1.818877102750140e-01     3.150387554686919e-01     1.093903749000560e-01
            3.349632505000791e-03     6.017352818211823e-01     8.960586792212499e-02
            3.454345984887935e-01     5.835045102907169e-01     8.090154301058200e-02
            1.386123033515377e-01     8.162351268485178e-01     5.466292404036400e-02
            -1.684976603227795e-01     8.253480847670428e-01     5.634122793587900e-02
            4.165386473699434e-01     7.902475552198603e-01     4.252012057136800e-02
            9.034867370714963e-02     9.613926190335345e-01     2.931195327772600e-02
            -2.455362814998164e-01     9.405839361024796e-01     2.467100183788900e-02
            4.132096241977805e-01     8.916824640844502e-01     1.620811375002300e-02
            -1.818877102750139e-01     3.150387554686920e-01     1.093903749000560e-01
            -5.194432241580319e-01     3.037685077532641e-01     8.960586792212499e-02
            -3.326124298901625e-01     5.909073927827312e-01     8.090154301058200e-02
            -6.375742036362613e-01     5.281593394037656e-01     5.466292404036400e-02
            -7.990212385344811e-01     2.667507880657530e-01     5.634122793587900e-02
            -4.761051344139732e-01     7.558568278903094e-01     4.252012057136800e-02
            -7.874160942403209e-01     5.589405561453900e-01     2.931195327772600e-02
            -9.373377238062147e-01     2.576513107216316e-01     2.467100183788900e-02
            -5.656148539073489e-01     8.036912636857241e-01     1.620811375002300e-02
            -3.637754205500280e-01     4.454964043850740e-17     1.093903749000560e-01
            -5.227928566630329e-01    -2.979667740679182e-01     8.960586792212499e-02
            -6.780470283789560e-01     7.402882492014264e-03     8.090154301058200e-02
            -7.761865069877990e-01    -2.880757874447524e-01     5.466292404036400e-02
            -6.305235782117017e-01    -5.585972967012895e-01     5.634122793587900e-02
            -8.926437817839168e-01    -3.439072732955099e-02     4.252012057136800e-02
            -8.777647679474704e-01    -4.024520628881447e-01     2.931195327772600e-02
            -6.918014423063982e-01    -6.829326253808480e-01     2.467100183788900e-02
            -9.788244781051294e-01    -8.799120039872614e-02     1.620811375002300e-02
            -1.818877102750142e-01    -3.150387554686919e-01     1.093903749000560e-01
            -3.349632505001132e-03    -6.017352818211823e-01     8.960586792212499e-02
            -3.454345984887936e-01    -5.835045102907169e-01     8.090154301058200e-02
            -1.386123033515380e-01    -8.162351268485178e-01     5.466292404036400e-02
            1.684976603227791e-01    -8.253480847670429e-01     5.634122793587900e-02
            -4.165386473699437e-01    -7.902475552198602e-01     4.252012057136800e-02
            -9.034867370714976e-02    -9.613926190335345e-01     2.931195327772600e-02
            2.455362814998162e-01    -9.405839361024796e-01     2.467100183788900e-02
            -4.132096241977808e-01    -8.916824640844501e-01     1.620811375002300e-02
            1.818877102750137e-01    -3.150387554686921e-01     1.093903749000560e-01
            5.194432241580317e-01    -3.037685077532644e-01     8.960586792212499e-02
            3.326124298901622e-01    -5.909073927827314e-01     8.090154301058200e-02
            6.375742036362610e-01    -5.281593394037659e-01     5.466292404036400e-02
            7.990212385344809e-01    -2.667507880657538e-01     5.634122793587900e-02
            4.761051344139728e-01    -7.558568278903097e-01     4.252012057136800e-02
            7.874160942403207e-01    -5.589405561453904e-01     2.931195327772600e-02
            9.373377238062146e-01    -2.576513107216322e-01     2.467100183788900e-02
            5.656148539073483e-01    -8.036912636857244e-01     1.620811375002300e-02];

    case 21

        xyw=[
        	 0.00000000000000000e+00 -0.00000000000000000e+00 8.25811521382980029e-02
         	 3.04013572681816002e-01 -0.00000000000000000e+00 7.75352258053950000e-02
         	 4.40694016196490179e-01 2.57132489832597622e-01 6.78774177433670023e-02
         	 5.79499647487903169e-01 4.88739353088766051e-03 6.34150063652629964e-02
         	 5.50105286219198408e-01 4.75114145349952555e-01 4.97857800200620024e-02
         	 6.92281228948658622e-01 2.43061166275965523e-01 5.07446041956680013e-02
         	 7.97440299007346587e-01 -1.24063798792593393e-03 4.27127158785090000e-02
         	 7.86373154240960215e-01 4.03579073034956715e-01 3.15690377282199994e-02
         	 6.50105371428008727e-01 5.99114391310717975e-01 2.84440998128859986e-02
         	 8.86200465223792255e-01 1.78919833611413004e-01 2.81712664421689997e-02
         	 9.29801038410373204e-01 -7.95095441200489722e-02 2.16943608871409989e-02
         	 7.75866334780149991e-01 5.92052910026407586e-01 1.52662780433670004e-02
         	 9.05435040441350369e-01 3.73112720135127751e-01 1.38525434538689998e-02
         	 9.75857344928653725e-01 1.20811768957808915e-01 1.13761699412430004e-02
         	 9.78835366622349046e-01 -1.47936496577614873e-01 7.39074392475800018e-03
         	 1.52006786340908029e-01 2.63283477037719482e-01 7.75352258053950000e-02
         	 -2.33626023512826424e-03 5.10218458238250183e-01 6.78774177433670023e-02
         	 2.85517216787911254e-01 5.04305112974095082e-01 6.34150063652629964e-02
         	 -1.36408276460791789e-01 7.13962225296911890e-01 4.97857800200620024e-02
         	 1.35643469805869721e-01 7.21063713970632181e-01 5.07446041956680013e-02
         	 3.99794573518117258e-01 6.89983237947857764e-01 4.27127158785090000e-02
         	 4.36768474364323117e-02 8.82808664944248700e-01 3.15690377282199994e-02
         	 -1.93795596933928310e-01 8.62564962448732708e-01 2.84440998128859986e-02
         	 2.88151111463527732e-01 8.56932032535098531e-01 2.81712664421689997e-02
         	 5.33757804256468793e-01 7.65476547668509144e-01 2.16943608871409989e-02
         	 -1.24799693077296311e-01 9.67946410873935736e-01 1.52662780433670004e-02
         	 1.29592426108541020e-01 9.70686106566363782e-01 1.38525434538689998e-02
         	 3.83302611470728138e-01 9.05523135656752021e-01 1.13761699412430004e-02
         	 6.17534447494258787e-01 7.73728045328801461e-01 7.39074392475800018e-03
         	 -1.52006786340907946e-01 2.63283477037719482e-01 7.75352258053950000e-02
         	 -4.43030276431618408e-01 2.53085968405652617e-01 6.78774177433670023e-02
         	 -2.93982430699992026e-01 4.99417719443207397e-01 6.34150063652629964e-02
         	 -6.86513562679990197e-01 2.38848079946959585e-01 4.97857800200620024e-02
         	 -5.56637759142788680e-01 4.78002547694666935e-01 5.07446041956680013e-02
         	 -3.97645725489229385e-01 6.91223875935783716e-01 4.27127158785090000e-02
         	 -7.42696306804527806e-01 4.79229591909291985e-01 3.15690377282199994e-02
         	 -8.43900968361937065e-01 2.63450571138014900e-01 2.84440998128859986e-02
         	 -5.98049353760264468e-01 6.78012198923685694e-01 2.81712664421689997e-02
         	 -3.96043234153904244e-01 8.44986091788558435e-01 2.16943608871409989e-02
         	 -9.00666027857446538e-01 3.75893500847528261e-01 1.52662780433670004e-02
         	 -7.75842614332809100e-01 5.97573386431236475e-01 1.38525434538689998e-02
         	 -5.92554733457925420e-01 7.84711366698943258e-01 1.13761699412430004e-02
         	 -3.61300919128090092e-01 9.21664541906416446e-01 7.39074392475800018e-03
         	 -3.04013572681816002e-01 3.72309248682137197e-17 7.75352258053950000e-02
         	 -4.40694016196490179e-01 -2.57132489832597566e-01 6.78774177433670023e-02
         	 -5.79499647487903169e-01 -4.88739353088769955e-03 6.34150063652629964e-02
         	 -5.50105286219198408e-01 -4.75114145349952555e-01 4.97857800200620024e-02
         	 -6.92281228948658733e-01 -2.43061166275965301e-01 5.07446041956680013e-02
         	 -7.97440299007346587e-01 1.24063798792592049e-03 4.27127158785090000e-02
         	 -7.86373154240960215e-01 -4.03579073034956659e-01 3.15690377282199994e-02
         	 -6.50105371428008838e-01 -5.99114391310717975e-01 2.84440998128859986e-02
         	 -8.86200465223792255e-01 -1.78919833611412921e-01 2.81712664421689997e-02
         	 -9.29801038410373204e-01 7.95095441200492081e-02 2.16943608871409989e-02
         	 -7.75866334780149991e-01 -5.92052910026407586e-01 1.52662780433670004e-02
         	 -9.05435040441350480e-01 -3.73112720135127474e-01 1.38525434538689998e-02
         	 -9.75857344928653725e-01 -1.20811768957808638e-01 1.13761699412430004e-02
         	 -9.78835366622349046e-01 1.47936496577615040e-01 7.39074392475800018e-03
         	 -1.52006786340908140e-01 -2.63283477037719427e-01 7.75352258053950000e-02
         	 2.33626023512808860e-03 -5.10218458238250183e-01 6.78774177433670023e-02
         	 -2.85517216787911532e-01 -5.04305112974094860e-01 6.34150063652629964e-02
         	 1.36408276460791705e-01 -7.13962225296911890e-01 4.97857800200620024e-02
         	 -1.35643469805870137e-01 -7.21063713970632181e-01 5.07446041956680013e-02
         	 -3.99794573518117369e-01 -6.89983237947857764e-01 4.27127158785090000e-02
         	 -4.36768474364330125e-02 -8.82808664944248589e-01 3.15690377282199994e-02
         	 1.93795596933928393e-01 -8.62564962448732708e-01 2.84440998128859986e-02
         	 -2.88151111463528342e-01 -8.56932032535098309e-01 2.81712664421689997e-02
         	 -5.33757804256469126e-01 -7.65476547668508922e-01 2.16943608871409989e-02
         	 1.24799693077296200e-01 -9.67946410873935736e-01 1.52662780433670004e-02
         	 -1.29592426108541547e-01 -9.70686106566363782e-01 1.38525434538689998e-02
         	 -3.83302611470728694e-01 -9.05523135656751799e-01 1.13761699412430004e-02
         	 -6.17534447494258787e-01 -7.73728045328801461e-01 7.39074392475800018e-03
         	 1.52006786340907807e-01 -2.63283477037719593e-01 7.75352258053950000e-02
         	 4.43030276431618297e-01 -2.53085968405652895e-01 6.78774177433670023e-02
         	 2.93982430699991470e-01 -4.99417719443207730e-01 6.34150063652629964e-02
         	 6.86513562679990197e-01 -2.38848079946959696e-01 4.97857800200620024e-02
         	 5.56637759142788457e-01 -4.78002547694667268e-01 5.07446041956680013e-02
         	 3.97645725489228996e-01 -6.91223875935784049e-01 4.27127158785090000e-02
         	 7.42696306804527473e-01 -4.79229591909292763e-01 3.15690377282199994e-02
         	 8.43900968361936732e-01 -2.63450571138015732e-01 2.84440998128859986e-02
         	 5.98049353760263802e-01 -6.78012198923686360e-01 2.81712664421689997e-02
         	 3.96043234153903745e-01 -8.44986091788558658e-01 2.16943608871409989e-02
         	 9.00666027857446427e-01 -3.75893500847528705e-01 1.52662780433670004e-02
         	 7.75842614332808878e-01 -5.97573386431236808e-01 1.38525434538689998e-02
         	 5.92554733457924976e-01 -7.84711366698943591e-01 1.13761699412430004e-02
         	 3.61300919128089981e-01 -9.21664541906416557e-01 7.39074392475800018e-03
         	 ];


    case 25

        xyw=[
            0.00000000000000000e+00 -0.00000000000000000e+00 6.02856549639200004e-02
            2.25532240989104765e-01 -1.30537147540723514e-01 5.75564279322630032e-02
            -2.70353364974284344e-14 -2.61434390723290999e-01 5.76849776494239991e-02
            4.41030807710878991e-01 -0.00000000000000000e+00 5.22532741291220001e-02
            2.20552280032051379e-01 -3.83127010340809759e-01 5.24533182837680009e-02
            4.35283827703710657e-01 -2.51606690838801328e-01 4.95549774394530018e-02
            -5.22890631176518975e-14 -5.05640436876208965e-01 5.02989555370279978e-02
            6.30829911180354252e-01 -1.21397737902834466e-01 4.24669514063149983e-02
            4.25122391290624291e-01 -4.83342558511969778e-01 4.21671800188630000e-02
            2.17561113953971930e-01 -6.07275599463581228e-01 4.31938694256460004e-02
            6.16718181354638739e-01 -3.51813776938591782e-01 3.67610926125130008e-02
            -7.34571952228183226e-14 -7.10338378038905982e-01 3.92341550411079987e-02
            7.92191541262275978e-01 -0.00000000000000000e+00 3.29972971413979993e-02
            4.30866917881915890e-01 -6.69978233454118222e-01 3.28077062322240029e-02
            2.23915587897890678e-01 -7.80135722670816323e-01 3.19544797103510012e-02
            7.81392013948421726e-01 -2.31917588655487034e-01 3.04675075414049985e-02
            6.13132882642399846e-01 -5.38200687382084264e-01 2.76291778407050000e-02
            -8.87129390908551576e-14 -8.57862937370733047e-01 2.70818077350040015e-02
            7.66877214617616265e-01 -4.29961578398521449e-01 2.24899281870759998e-02
            9.06663702168989172e-01 -1.09773328698934483e-01 2.03578719941309993e-02
            4.32210260034033789e-01 -8.08071480750580329e-01 2.22177384194520017e-02
            6.22377653374615480e-01 -6.73466596156606090e-01 2.06011463327130002e-02
            2.16690426201874536e-01 -8.99525107585569583e-01 1.93980940509050001e-02
            8.88999724156006588e-01 -3.08128258351988815e-01 1.57369993317079994e-02
            -9.84527088625343965e-14 -9.52047479008906961e-01 1.43538449317160005e-02
            7.83339595266909328e-01 -5.48290069961165805e-01 1.43298182731320008e-02
            9.79515608138995963e-01 -0.00000000000000000e+00 8.69040330920200013e-03
            6.30222402475750121e-01 -7.54274645002916300e-01 9.36082186783200078e-03
            2.68976322370745535e-01 -9.46748327882029894e-01 7.30804722908900018e-03
            4.53089257209643270e-01 -8.74264595703291514e-01 8.30233571477900084e-03
            9.70752122589158994e-01 -1.82275162307021482e-01 6.94968127452700018e-03
            9.02393562597139653e-01 -4.04155832650167857e-01 6.78725489574400018e-03
            1.00609987961041716e-01 -9.86291962696266067e-01 4.35905025012500030e-03
            7.83460871001050219e-01 -6.13251747023185190e-01 3.81791565474800003e-03
            2.25532240989104765e-01 1.30537147540723514e-01 5.75564279322630032e-02
            -2.70353364974284344e-14 2.61434390723290999e-01 5.76849776494239991e-02
            2.20552280032051379e-01 3.83127010340809759e-01 5.24533182837680009e-02
            4.35283827703710657e-01 2.51606690838801328e-01 4.95549774394530018e-02
            -5.22890631176518975e-14 5.05640436876208965e-01 5.02989555370279978e-02
            6.30829911180354252e-01 1.21397737902834466e-01 4.24669514063149983e-02
            4.25122391290624291e-01 4.83342558511969778e-01 4.21671800188630000e-02
            2.17561113953971930e-01 6.07275599463581228e-01 4.31938694256460004e-02
            6.16718181354638739e-01 3.51813776938591782e-01 3.67610926125130008e-02
            -7.34571952228183226e-14 7.10338378038905982e-01 3.92341550411079987e-02
            4.30866917881915890e-01 6.69978233454118222e-01 3.28077062322240029e-02
            2.23915587897890678e-01 7.80135722670816323e-01 3.19544797103510012e-02
            7.81392013948421726e-01 2.31917588655487034e-01 3.04675075414049985e-02
            6.13132882642399846e-01 5.38200687382084264e-01 2.76291778407050000e-02
            -8.87129390908551576e-14 8.57862937370733047e-01 2.70818077350040015e-02
            7.66877214617616265e-01 4.29961578398521449e-01 2.24899281870759998e-02
            9.06663702168989172e-01 1.09773328698934483e-01 2.03578719941309993e-02
            4.32210260034033789e-01 8.08071480750580329e-01 2.22177384194520017e-02
            6.22377653374615480e-01 6.73466596156606090e-01 2.06011463327130002e-02
            2.16690426201874536e-01 8.99525107585569583e-01 1.93980940509050001e-02
            8.88999724156006588e-01 3.08128258351988815e-01 1.57369993317079994e-02
            -9.84527088625343965e-14 9.52047479008906961e-01 1.43538449317160005e-02
            7.83339595266909328e-01 5.48290069961165805e-01 1.43298182731320008e-02
            6.30222402475750121e-01 7.54274645002916300e-01 9.36082186783200078e-03
            2.68976322370745535e-01 9.46748327882029894e-01 7.30804722908900018e-03
            4.53089257209643270e-01 8.74264595703291514e-01 8.30233571477900084e-03
            9.70752122589158994e-01 1.82275162307021482e-01 6.94968127452700018e-03
            9.02393562597139653e-01 4.04155832650167857e-01 6.78725489574400018e-03
            1.00609987961041716e-01 9.86291962696266067e-01 4.35905025012500030e-03
            7.83460871001050219e-01 6.13251747023185190e-01 3.81791565474800003e-03
            -2.25532240989104765e-01 1.30537147540723542e-01 5.75564279322630032e-02
            -4.41030807710878991e-01 5.40106966988499784e-17 5.22532741291220001e-02
            -2.20552280032051240e-01 3.83127010340809870e-01 5.24533182837680009e-02
            -4.35283827703710546e-01 2.51606690838801439e-01 4.95549774394530018e-02
            -6.30829911180354252e-01 1.21397737902834590e-01 4.24669514063149983e-02
            -4.25122391290624291e-01 4.83342558511969778e-01 4.21671800188630000e-02
            -2.17561113953971846e-01 6.07275599463581339e-01 4.31938694256460004e-02
            -6.16718181354638739e-01 3.51813776938591727e-01 3.67610926125130008e-02
            -7.92191541262275978e-01 9.70154835318454681e-17 3.29972971413979993e-02
            -4.30866917881916001e-01 6.69978233454118111e-01 3.28077062322240029e-02
            -2.23915587897890594e-01 7.80135722670816323e-01 3.19544797103510012e-02
            -7.81392013948421615e-01 2.31917588655487311e-01 3.04675075414049985e-02
            -6.13132882642399735e-01 5.38200687382084375e-01 2.76291778407050000e-02
            -7.66877214617616154e-01 4.29961578398521671e-01 2.24899281870759998e-02
            -9.06663702168989172e-01 1.09773328698934636e-01 2.03578719941309993e-02
            -4.32210260034033900e-01 8.08071480750580329e-01 2.22177384194520017e-02
            -6.22377653374615258e-01 6.73466596156606312e-01 2.06011463327130002e-02
            -2.16690426201874425e-01 8.99525107585569583e-01 1.93980940509050001e-02
            -8.88999724156006588e-01 3.08128258351988871e-01 1.57369993317079994e-02
            -7.83339595266909439e-01 5.48290069961165805e-01 1.43298182731320008e-02
            -9.79515608138995963e-01 1.19956065422229442e-16 8.69040330920200013e-03
            -6.30222402475750121e-01 7.54274645002916189e-01 9.36082186783200078e-03
            -2.68976322370745424e-01 9.46748327882029894e-01 7.30804722908900018e-03
            -4.53089257209643159e-01 8.74264595703291625e-01 8.30233571477900084e-03
            -9.70752122589158994e-01 1.82275162307021538e-01 6.94968127452700018e-03
            -9.02393562597139542e-01 4.04155832650168190e-01 6.78725489574400018e-03
            -1.00609987961041591e-01 9.86291962696266067e-01 4.35905025012500030e-03
            -7.83460871001049997e-01 6.13251747023185412e-01 3.81791565474800003e-03
            -2.25532240989104793e-01 -1.30537147540723486e-01 5.75564279322630032e-02
            -2.20552280032051323e-01 -3.83127010340809815e-01 5.24533182837680009e-02
            -4.35283827703710602e-01 -2.51606690838801328e-01 4.95549774394530018e-02
            -6.30829911180354252e-01 -1.21397737902834438e-01 4.24669514063149983e-02
            -4.25122391290624457e-01 -4.83342558511969611e-01 4.21671800188630000e-02
            -2.17561113953971846e-01 -6.07275599463581339e-01 4.31938694256460004e-02
            -6.16718181354638850e-01 -3.51813776938591616e-01 3.67610926125130008e-02
            -4.30866917881916167e-01 -6.69978233454118000e-01 3.28077062322240029e-02
            -2.23915587897890761e-01 -7.80135722670816323e-01 3.19544797103510012e-02
            -7.81392013948421726e-01 -2.31917588655487117e-01 3.04675075414049985e-02
            -6.13132882642399846e-01 -5.38200687382084153e-01 2.76291778407050000e-02
            -7.66877214617616265e-01 -4.29961578398521505e-01 2.24899281870759998e-02
            -9.06663702168989172e-01 -1.09773328698934428e-01 2.03578719941309993e-02
            -4.32210260034034066e-01 -8.08071480750580329e-01 2.22177384194520017e-02
            -6.22377653374615480e-01 -6.73466596156606201e-01 2.06011463327130002e-02
            -2.16690426201874620e-01 -8.99525107585569583e-01 1.93980940509050001e-02
            -8.88999724156006588e-01 -3.08128258351988649e-01 1.57369993317079994e-02
            -7.83339595266909550e-01 -5.48290069961165694e-01 1.43298182731320008e-02
            -6.30222402475750343e-01 -7.54274645002916078e-01 9.36082186783200078e-03
            -2.68976322370746090e-01 -9.46748327882029672e-01 7.30804722908900018e-03
            -4.53089257209643770e-01 -8.74264595703291292e-01 8.30233571477900084e-03
            -9.70752122589159105e-01 -1.82275162307021288e-01 6.94968127452700018e-03
            -9.02393562597139653e-01 -4.04155832650167968e-01 6.78725489574400018e-03
            -1.00609987961042271e-01 -9.86291962696265956e-01 4.35905025012500030e-03
            -7.83460871001050108e-01 -6.13251747023185190e-01 3.81791565474800003e-03
            ];


    case 27

        xyw=[
            -0.00000000000000000e+00 -0.00000000000000000e+00 5.09274859060920015e-02
            2.34798437595721998e-01 -0.00000000000000000e+00 4.84484274382959990e-02
            1.18527563304572509e-01 -2.12112222677787621e-01 4.93046934778990018e-02
            3.49545320057419673e-01 -2.03186502878094394e-01 4.47401181387300020e-02
            -4.30973325910068355e-14 -4.16755489202089990e-01 4.61751204783300007e-02
            4.54247701342346988e-01 -0.00000000000000000e+00 4.01030413076060022e-02
            2.34347803916962305e-01 -4.08058686312712526e-01 4.42036489104759983e-02
            5.71054346246281930e-01 -1.57490107606332475e-01 3.75673065655849972e-02
            4.66812545003621127e-01 -3.66273957912282677e-01 3.93043278698939971e-02
            1.15070437048947441e-01 -5.98771465988003015e-01 3.95968303194259991e-02
            3.50256031241477306e-01 -5.60503198966106342e-01 3.62617028121470023e-02
            6.89924349020905048e-01 -0.00000000000000000e+00 3.20778937501840014e-02
            6.62468128759030495e-01 -3.16329042233677638e-01 3.07392893480929984e-02
            5.59508669766867861e-01 -5.17931685643600703e-01 3.11466284382460011e-02
            2.27303827938603903e-01 -7.29229923808540126e-01 2.83779880998580002e-02
            -7.90960089347592130e-14 -7.64866267023147972e-01 3.33890460299159988e-02
            7.71506683708399477e-01 -1.56989597523379626e-01 2.57083842639219996e-02
            4.29121468354520752e-01 -6.95344931896875851e-01 2.58360206195830007e-02
            7.39322106461698936e-01 -4.40318540572500916e-01 2.40318717209190014e-02
            8.62369699060634987e-01 -0.00000000000000000e+00 2.35167461187659986e-02
            2.06535112058564146e-01 -8.49669367006892551e-01 2.17758417036949987e-02
            8.46614772832192775e-01 -2.63480855585610796e-01 1.97961429878300010e-02
            6.16837224591462063e-01 -6.40119431502210512e-01 2.17600034044280015e-02
            4.37127219525502086e-12 -8.92717832782067955e-01 1.06559676971380007e-02
            4.18071965684947144e-01 -8.11613474197125040e-01 1.87604587938110003e-02
            9.35288265084122883e-01 -1.21492677910526634e-01 1.53674474600070006e-02
            7.70216302783209850e-01 -5.50533599817005448e-01 1.38469902725390000e-02
            2.56359261017966467e-01 -9.20349279469361514e-01 1.10511462845039998e-02
            8.80988117464571951e-01 -3.75522942843591612e-01 1.27858939096409992e-02
            5.92198975233497849e-01 -7.56439048370652456e-01 1.29864624410900003e-02
            9.57200583817352235e-02 -9.62612009551997772e-01 9.93838116299100051e-03
            4.24612406653264218e-01 -8.86722302679245566e-01 8.38781146725399961e-03
            7.41945620005673212e-01 -6.51342064152613864e-01 6.69118144802700009e-03
            9.88853072566576952e-01 0.00000000000000000e+00 6.79637524734000038e-03
            9.60695491188082062e-01 -2.36588099524980527e-01 6.47808710837699964e-03
            8.75034635124043891e-01 -4.73445959465637611e-01 4.09617621255200029e-03
            2.38749117935430105e-01 -9.69074226430318819e-01 2.81852571362999982e-03
            -1.03328384762395998e-13 -9.99195749509101949e-01 2.27905775755400013e-03
            5.93804365445971261e-01 -8.03846880987796775e-01 2.25810920463400018e-03
            1.18527563304572509e-01 2.12112222677787621e-01 4.93046934778990018e-02
            3.49545320057419673e-01 2.03186502878094394e-01 4.47401181387300020e-02
            -4.30973325910068355e-14 4.16755489202089990e-01 4.61751204783300007e-02
            2.34347803916962305e-01 4.08058686312712526e-01 4.42036489104759983e-02
            5.71054346246281930e-01 1.57490107606332475e-01 3.75673065655849972e-02
            4.66812545003621127e-01 3.66273957912282677e-01 3.93043278698939971e-02
            1.15070437048947441e-01 5.98771465988003015e-01 3.95968303194259991e-02
            3.50256031241477306e-01 5.60503198966106342e-01 3.62617028121470023e-02
            6.62468128759030495e-01 3.16329042233677638e-01 3.07392893480929984e-02
            5.59508669766867861e-01 5.17931685643600703e-01 3.11466284382460011e-02
            2.27303827938603903e-01 7.29229923808540126e-01 2.83779880998580002e-02
            -7.90960089347592130e-14 7.64866267023147972e-01 3.33890460299159988e-02
            7.71506683708399477e-01 1.56989597523379626e-01 2.57083842639219996e-02
            4.29121468354520752e-01 6.95344931896875851e-01 2.58360206195830007e-02
            7.39322106461698936e-01 4.40318540572500916e-01 2.40318717209190014e-02
            2.06535112058564146e-01 8.49669367006892551e-01 2.17758417036949987e-02
            8.46614772832192775e-01 2.63480855585610796e-01 1.97961429878300010e-02
            6.16837224591462063e-01 6.40119431502210512e-01 2.17600034044280015e-02
            4.37127219525502086e-12 8.92717832782067955e-01 1.06559676971380007e-02
            4.18071965684947144e-01 8.11613474197125040e-01 1.87604587938110003e-02
            9.35288265084122883e-01 1.21492677910526634e-01 1.53674474600070006e-02
            7.70216302783209850e-01 5.50533599817005448e-01 1.38469902725390000e-02
            2.56359261017966467e-01 9.20349279469361514e-01 1.10511462845039998e-02
            8.80988117464571951e-01 3.75522942843591612e-01 1.27858939096409992e-02
            5.92198975233497849e-01 7.56439048370652456e-01 1.29864624410900003e-02
            9.57200583817352235e-02 9.62612009551997772e-01 9.93838116299100051e-03
            4.24612406653264218e-01 8.86722302679245566e-01 8.38781146725399961e-03
            7.41945620005673212e-01 6.51342064152613864e-01 6.69118144802700009e-03
            9.60695491188082062e-01 2.36588099524980527e-01 6.47808710837699964e-03
            8.75034635124043891e-01 4.73445959465637611e-01 4.09617621255200029e-03
            2.38749117935430105e-01 9.69074226430318819e-01 2.81852571362999982e-03
            -1.03328384762395998e-13 9.99195749509101949e-01 2.27905775755400013e-03
            5.93804365445971261e-01 8.03846880987796775e-01 2.25810920463400018e-03
            -2.34798437595721998e-01 2.87545155046400520e-17 4.84484274382959990e-02
            -1.18527563304572439e-01 2.12112222677787676e-01 4.93046934778990018e-02
            -3.49545320057419617e-01 2.03186502878094449e-01 4.47401181387300020e-02
            -4.54247701342346988e-01 5.56292993468948153e-17 4.01030413076060022e-02
            -2.34347803916962361e-01 4.08058686312712471e-01 4.42036489104759983e-02
            -5.71054346246281930e-01 1.57490107606332447e-01 3.75673065655849972e-02
            -4.66812545003621127e-01 3.66273957912282677e-01 3.93043278698939971e-02
            -1.15070437048947372e-01 5.98771465988003015e-01 3.95968303194259991e-02
            -3.50256031241477361e-01 5.60503198966106231e-01 3.62617028121470023e-02
            -6.89924349020905048e-01 8.44913645682272698e-17 3.20778937501840014e-02
            -6.62468128759030495e-01 3.16329042233677638e-01 3.07392893480929984e-02
            -5.59508669766867528e-01 5.17931685643600814e-01 3.11466284382460011e-02
            -2.27303827938603820e-01 7.29229923808540126e-01 2.83779880998580002e-02
            -7.71506683708399477e-01 1.56989597523379543e-01 2.57083842639219996e-02
            -4.29121468354520641e-01 6.95344931896875851e-01 2.58360206195830007e-02
            -7.39322106461698936e-01 4.40318540572501027e-01 2.40318717209190014e-02
            -8.62369699060634987e-01 1.05609829163627295e-16 2.35167461187659986e-02
            -2.06535112058564008e-01 8.49669367006892662e-01 2.17758417036949987e-02
            -8.46614772832192775e-01 2.63480855585610796e-01 1.97961429878300010e-02
            -6.16837224591462063e-01 6.40119431502210512e-01 2.17600034044280015e-02
            -4.37116286885137523e-12 8.92717832782067955e-01 1.06559676971380007e-02
            -4.18071965684946811e-01 8.11613474197125151e-01 1.87604587938110003e-02
            -9.35288265084122883e-01 1.21492677910526800e-01 1.53674474600070006e-02
            -7.70216302783209850e-01 5.50533599817005448e-01 1.38469902725390000e-02
            -2.56359261017966356e-01 9.20349279469361514e-01 1.10511462845039998e-02
            -8.80988117464571951e-01 3.75522942843591723e-01 1.27858939096409992e-02
            -5.92198975233497515e-01 7.56439048370652678e-01 1.29864624410900003e-02
            -9.57200583817350986e-02 9.62612009551997772e-01 9.93838116299100051e-03
            -4.24612406653263885e-01 8.86722302679245677e-01 8.38781146725399961e-03
            -7.41945620005673101e-01 6.51342064152613864e-01 6.69118144802700009e-03
            -9.88853072566576952e-01 1.21099575014568376e-16 6.79637524734000038e-03
            -9.60695491188081951e-01 2.36588099524980860e-01 6.47808710837699964e-03
            -8.75034635124044002e-01 4.73445959465637556e-01 4.09617621255200029e-03
            -2.38749117935429994e-01 9.69074226430318819e-01 2.81852571362999982e-03
            -5.93804365445970928e-01 8.03846880987796886e-01 2.25810920463400018e-03
            -1.18527563304572495e-01 -2.12112222677787648e-01 4.93046934778990018e-02
            -3.49545320057419673e-01 -2.03186502878094366e-01 4.47401181387300020e-02
            -2.34347803916962472e-01 -4.08058686312712415e-01 4.42036489104759983e-02
            -5.71054346246282041e-01 -1.57490107606332308e-01 3.75673065655849972e-02
            -4.66812545003621182e-01 -3.66273957912282511e-01 3.93043278698939971e-02
            -1.15070437048947247e-01 -5.98771465988003015e-01 3.95968303194259991e-02
            -3.50256031241477528e-01 -5.60503198966106231e-01 3.62617028121470023e-02
            -6.62468128759030606e-01 -3.16329042233677471e-01 3.07392893480929984e-02
            -5.59508669766867750e-01 -5.17931685643600814e-01 3.11466284382460011e-02
            -2.27303827938604180e-01 -7.29229923808540015e-01 2.83779880998580002e-02
            -7.71506683708399477e-01 -1.56989597523379348e-01 2.57083842639219996e-02
            -4.29121468354521085e-01 -6.95344931896875629e-01 2.58360206195830007e-02
            -7.39322106461699047e-01 -4.40318540572500861e-01 2.40318717209190014e-02
            -2.06535112058563869e-01 -8.49669367006892662e-01 2.17758417036949987e-02
            -8.46614772832192775e-01 -2.63480855585610574e-01 1.97961429878300010e-02
            -6.16837224591462285e-01 -6.40119431502210401e-01 2.17600034044280015e-02
            -4.37138152165866649e-12 -8.92717832782067955e-01 1.06559676971380007e-02
            -4.18071965684947033e-01 -8.11613474197125040e-01 1.87604587938110003e-02
            -9.35288265084122883e-01 -1.21492677910526578e-01 1.53674474600070006e-02
            -7.70216302783210072e-01 -5.50533599817005337e-01 1.38469902725390000e-02
            -2.56359261017966578e-01 -9.20349279469361514e-01 1.10511462845039998e-02
            -8.80988117464572063e-01 -3.75522942843591556e-01 1.27858939096409992e-02
            -5.92198975233497737e-01 -7.56439048370652567e-01 1.29864624410900003e-02
            -9.57200583817355566e-02 -9.62612009551997772e-01 9.93838116299100051e-03
            -4.24612406653264107e-01 -8.86722302679245566e-01 8.38781146725399961e-03
            -7.41945620005673323e-01 -6.51342064152613753e-01 6.69118144802700009e-03
            -9.60695491188081951e-01 -2.36588099524980638e-01 6.47808710837699964e-03
            -8.75034635124044113e-01 -4.73445959465637334e-01 4.09617621255200029e-03
            -2.38749117935430022e-01 -9.69074226430318819e-01 2.81852571362999982e-03
            -5.93804365445971150e-01 -8.03846880987796775e-01 2.25810920463400018e-03
            ];


    case 29

        xyw=[
            0.00000000000000000e+00 0.00000000000000000e+00 4.59105948379200032e-02
            1.97126058043436164e-01 -1.14498125129436260e-01 4.43505599351299989e-02
            -2.36657355047565625e-14 -2.28850014250537998e-01 4.43188127572739995e-02
            3.87373019515326988e-01 0.00000000000000000e+00 4.12923342929980008e-02
            1.94287645215130922e-01 -3.37199653964403268e-01 4.12615869499059970e-02
            3.84490036075814390e-01 -2.22864704039941242e-01 3.98391324131359970e-02
            -4.59964263793746440e-14 -4.44790014249843990e-01 3.95976752447999986e-02
            5.61438118686107490e-01 -1.07808699315683182e-01 3.54700769128179980e-02
            1.85472952693121529e-01 -5.42411606479276887e-01 3.52132199982249997e-02
            3.72864861941384074e-01 -4.35648368982446188e-01 3.55369330489620008e-02
            5.49198477504229565e-01 -3.20508174909668653e-01 3.27302571500259973e-02
            -6.59325775315276365e-14 -6.37574577161657996e-01 3.19259440975920017e-02
            7.17395704411209034e-01 -0.00000000000000000e+00 2.88768698167639996e-02
            3.46014822332797123e-01 -6.28911301470124195e-01 2.89341807847219995e-02
            7.06162441883089298e-01 -2.03245940703242256e-01 2.71300599470480006e-02
            5.20957967689390844e-01 -5.22013467876470072e-01 2.78364384415279997e-02
            1.64379043259049090e-01 -7.22167528201810960e-01 2.69711169189679988e-02
            6.76689308430332814e-01 -4.04148957540314602e-01 2.35353097682730016e-02
            -8.30998630098813601e-14 -8.03583933835768982e-01 2.25304717521520002e-02
            3.29086466960759327e-01 -7.77606353263585737e-01 2.07908584430229991e-02
            8.39604370312375892e-01 -9.59633257498410386e-02 2.07643442981759983e-02
            5.03011982619586906e-01 -6.81950202333370781e-01 2.17823354278430005e-02
            8.04395611155360069e-01 -2.84601425075108272e-01 1.64581882251110014e-02
            6.64100785284903017e-01 -5.63014994598688090e-01 1.95470296800790005e-02
            1.65763558303918229e-01 -8.56512875359185410e-01 1.75810605266129988e-02
            8.01365626727514657e-01 -4.21999252187156060e-01 1.46041409461930005e-02
            -9.44893250832169983e-14 -9.13721163978687967e-01 1.45419686877520001e-02
            9.03899601697777055e-01 -2.10741867015206724e-01 1.32049083995690007e-02
            4.29559531738208655e-01 -8.29200604841968447e-01 1.48775489232719996e-02
            9.36667832134787992e-01 -0.00000000000000000e+00 1.56166682709220003e-02
            6.03451305887448886e-01 -7.20913893672050166e-01 1.41200580534259993e-02
            2.51445644344053476e-01 -9.10200010223125022e-01 1.20285396158749999e-02
            7.55485127097505038e-01 -5.79993304544451194e-01 1.16609965287769995e-02
            8.80664217037948283e-01 -3.93835130829358115e-01 9.23194659013699959e-03
            9.64790530175848038e-02 -9.61529148213769669e-01 7.54879487187600038e-03
            9.46431560101082092e-01 -2.16117748398471560e-01 4.81966303270300010e-03
            3.78994297718384332e-01 -9.11907632977758786e-01 6.04849328176599969e-03
            5.50410264321637865e-01 -8.20157914295179036e-01 6.27262037489299991e-03
            9.83742136950106816e-01 -9.80814431115873825e-02 5.67603019140699994e-03
            7.08578429435026025e-01 -6.91283072334442861e-01 5.45762936751100023e-03
            8.43178295676352940e-01 -5.22267068192563300e-01 4.64472937826799961e-03
            2.03760426411230822e-01 -9.71318299234273042e-01 3.80009516747599994e-03
            -1.02822505495192837e-13 -9.94303846720565043e-01 3.15267603295200016e-03
            9.43651364587544506e-01 -3.15621683647809237e-01 3.26492061862900017e-03
            1.97126058043436164e-01 1.14498125129436260e-01 4.43505599351299989e-02
            -2.36657355047565625e-14 2.28850014250537998e-01 4.43188127572739995e-02
            1.94287645215130922e-01 3.37199653964403268e-01 4.12615869499059970e-02
            3.84490036075814390e-01 2.22864704039941242e-01 3.98391324131359970e-02
            -4.59964263793746440e-14 4.44790014249843990e-01 3.95976752447999986e-02
            5.61438118686107490e-01 1.07808699315683182e-01 3.54700769128179980e-02
            1.85472952693121529e-01 5.42411606479276887e-01 3.52132199982249997e-02
            3.72864861941384074e-01 4.35648368982446188e-01 3.55369330489620008e-02
            5.49198477504229565e-01 3.20508174909668653e-01 3.27302571500259973e-02
            -6.59325775315276365e-14 6.37574577161657996e-01 3.19259440975920017e-02
            3.46014822332797123e-01 6.28911301470124195e-01 2.89341807847219995e-02
            7.06162441883089298e-01 2.03245940703242256e-01 2.71300599470480006e-02
            5.20957967689390844e-01 5.22013467876470072e-01 2.78364384415279997e-02
            1.64379043259049090e-01 7.22167528201810960e-01 2.69711169189679988e-02
            6.76689308430332814e-01 4.04148957540314602e-01 2.35353097682730016e-02
            -8.30998630098813601e-14 8.03583933835768982e-01 2.25304717521520002e-02
            3.29086466960759327e-01 7.77606353263585737e-01 2.07908584430229991e-02
            8.39604370312375892e-01 9.59633257498410386e-02 2.07643442981759983e-02
            5.03011982619586906e-01 6.81950202333370781e-01 2.17823354278430005e-02
            8.04395611155360069e-01 2.84601425075108272e-01 1.64581882251110014e-02
            6.64100785284903017e-01 5.63014994598688090e-01 1.95470296800790005e-02
            1.65763558303918229e-01 8.56512875359185410e-01 1.75810605266129988e-02
            8.01365626727514657e-01 4.21999252187156060e-01 1.46041409461930005e-02
            -9.44893250832169983e-14 9.13721163978687967e-01 1.45419686877520001e-02
            9.03899601697777055e-01 2.10741867015206724e-01 1.32049083995690007e-02
            4.29559531738208655e-01 8.29200604841968447e-01 1.48775489232719996e-02
            6.03451305887448886e-01 7.20913893672050166e-01 1.41200580534259993e-02
            2.51445644344053476e-01 9.10200010223125022e-01 1.20285396158749999e-02
            7.55485127097505038e-01 5.79993304544451194e-01 1.16609965287769995e-02
            8.80664217037948283e-01 3.93835130829358115e-01 9.23194659013699959e-03
            9.64790530175848038e-02 9.61529148213769669e-01 7.54879487187600038e-03
            9.46431560101082092e-01 2.16117748398471560e-01 4.81966303270300010e-03
            3.78994297718384332e-01 9.11907632977758786e-01 6.04849328176599969e-03
            5.50410264321637865e-01 8.20157914295179036e-01 6.27262037489299991e-03
            9.83742136950106816e-01 9.80814431115873825e-02 5.67603019140699994e-03
            7.08578429435026025e-01 6.91283072334442861e-01 5.45762936751100023e-03
            8.43178295676352940e-01 5.22267068192563300e-01 4.64472937826799961e-03
            2.03760426411230822e-01 9.71318299234273042e-01 3.80009516747599994e-03
            -1.02822505495192837e-13 9.94303846720565043e-01 3.15267603295200016e-03
            9.43651364587544506e-01 3.15621683647809237e-01 3.26492061862900017e-03
            -1.97126058043436192e-01 1.14498125129436246e-01 4.43505599351299989e-02
            -3.87373019515326988e-01 4.74395128425490386e-17 4.12923342929980008e-02
            -1.94287645215130922e-01 3.37199653964403212e-01 4.12615869499059970e-02
            -3.84490036075814445e-01 2.22864704039941186e-01 3.98391324131359970e-02
            -5.61438118686107490e-01 1.07808699315683196e-01 3.54700769128179980e-02
            -1.85472952693121473e-01 5.42411606479276887e-01 3.52132199982249997e-02
            -3.72864861941383907e-01 4.35648368982446299e-01 3.55369330489620008e-02
            -5.49198477504229565e-01 3.20508174909668653e-01 3.27302571500259973e-02
            -7.17395704411209034e-01 8.78556353129247856e-17 2.88768698167639996e-02
            -3.46014822332797123e-01 6.28911301470124195e-01 2.89341807847219995e-02
            -7.06162441883089187e-01 2.03245940703242478e-01 2.71300599470480006e-02
            -5.20957967689390844e-01 5.22013467876470072e-01 2.78364384415279997e-02
            -1.64379043259048979e-01 7.22167528201811071e-01 2.69711169189679988e-02
            -6.76689308430332703e-01 4.04148957540314713e-01 2.35353097682730016e-02
            -3.29086466960759272e-01 7.77606353263585848e-01 2.07908584430229991e-02
            -8.39604370312375892e-01 9.59633257498410663e-02 2.07643442981759983e-02
            -5.03011982619586795e-01 6.81950202333370781e-01 2.17823354278430005e-02
            -8.04395611155360069e-01 2.84601425075108383e-01 1.64581882251110014e-02
            -6.64100785284903017e-01 5.63014994598688090e-01 1.95470296800790005e-02
            -1.65763558303918118e-01 8.56512875359185410e-01 1.75810605266129988e-02
            -8.01365626727514657e-01 4.21999252187156060e-01 1.46041409461930005e-02
            -9.03899601697777055e-01 2.10741867015206724e-01 1.32049083995690007e-02
            -4.29559531738208711e-01 8.29200604841968336e-01 1.48775489232719996e-02
            -9.36667832134787992e-01 1.14708726248815848e-16 1.56166682709220003e-02
            -6.03451305887448664e-01 7.20913893672050277e-01 1.41200580534259993e-02
            -2.51445644344053365e-01 9.10200010223125133e-01 1.20285396158749999e-02
            -7.55485127097505038e-01 5.79993304544451305e-01 1.16609965287769995e-02
            -8.80664217037948172e-01 3.93835130829358282e-01 9.23194659013699959e-03
            -9.64790530175846928e-02 9.61529148213769669e-01 7.54879487187600038e-03
            -9.46431560101081981e-01 2.16117748398471865e-01 4.81966303270300010e-03
            -3.78994297718384221e-01 9.11907632977758786e-01 6.04849328176599969e-03
            -5.50410264321637643e-01 8.20157914295179147e-01 6.27262037489299991e-03
            -9.83742136950106816e-01 9.80814431115876184e-02 5.67603019140699994e-03
            -7.08578429435025914e-01 6.91283072334442861e-01 5.45762936751100023e-03
            -8.43178295676352718e-01 5.22267068192563522e-01 4.64472937826799961e-03
            -2.03760426411230711e-01 9.71318299234273042e-01 3.80009516747599994e-03
            -9.43651364587544506e-01 3.15621683647809237e-01 3.26492061862900017e-03
            -1.97126058043436220e-01 -1.14498125129436190e-01 4.43505599351299989e-02
            -1.94287645215131033e-01 -3.37199653964403157e-01 4.12615869499059970e-02
            -3.84490036075814501e-01 -2.22864704039941075e-01 3.98391324131359970e-02
            -5.61438118686107490e-01 -1.07808699315683071e-01 3.54700769128179980e-02
            -1.85472952693121584e-01 -5.42411606479276776e-01 3.52132199982249997e-02
            -3.72864861941384018e-01 -4.35648368982446244e-01 3.55369330489620008e-02
            -5.49198477504229565e-01 -3.20508174909668542e-01 3.27302571500259973e-02
            -3.46014822332797289e-01 -6.28911301470124084e-01 2.89341807847219995e-02
            -7.06162441883089298e-01 -2.03245940703242312e-01 2.71300599470480006e-02
            -5.20957967689390955e-01 -5.22013467876469961e-01 2.78364384415279997e-02
            -1.64379043259049340e-01 -7.22167528201810960e-01 2.69711169189679988e-02
            -6.76689308430332814e-01 -4.04148957540314491e-01 2.35353097682730016e-02
            -3.29086466960759605e-01 -7.77606353263585626e-01 2.07908584430229991e-02
            -8.39604370312375892e-01 -9.59633257498408582e-02 2.07643442981759983e-02
            -5.03011982619587239e-01 -6.81950202333370448e-01 2.17823354278430005e-02
            -8.04395611155360180e-01 -2.84601425075108161e-01 1.64581882251110014e-02
            -6.64100785284903239e-01 -5.63014994598687868e-01 1.95470296800790005e-02
            -1.65763558303918146e-01 -8.56512875359185410e-01 1.75810605266129988e-02
            -8.01365626727514768e-01 -4.21999252187155838e-01 1.46041409461930005e-02
            -9.03899601697777055e-01 -2.10741867015206530e-01 1.32049083995690007e-02
            -4.29559531738208933e-01 -8.29200604841968225e-01 1.48775489232719996e-02
            -6.03451305887448886e-01 -7.20913893672050166e-01 1.41200580534259993e-02
            -2.51445644344054031e-01 -9.10200010223124911e-01 1.20285396158749999e-02
            -7.55485127097505149e-01 -5.79993304544451083e-01 1.16609965287769995e-02
            -8.80664217037948283e-01 -3.93835130829358060e-01 9.23194659013699959e-03
            -9.64790530175847205e-02 -9.61529148213769669e-01 7.54879487187600038e-03
            -9.46431560101081981e-01 -2.16117748398471615e-01 4.81966303270300010e-03
            -3.78994297718384443e-01 -9.11907632977758675e-01 6.04849328176599969e-03
            -5.50410264321638198e-01 -8.20157914295178703e-01 6.27262037489299991e-03
            -9.83742136950106816e-01 -9.80814431115873825e-02 5.67603019140699994e-03
            -7.08578429435026136e-01 -6.91283072334442750e-01 5.45762936751100023e-03
            -8.43178295676352940e-01 -5.22267068192563411e-01 4.64472937826799961e-03
            -2.03760426411231155e-01 -9.71318299234272930e-01 3.80009516747599994e-03
            -9.43651364587544617e-01 -3.15621683647809015e-01 3.26492061862900017e-03
            ];

    case 33

        xyw=[
            0.00000000000000000e+00 0.00000000000000000e+00 3.62134640499680027e-02
            2.01958986148607994e-01 -0.00000000000000000e+00 3.51437839711939981e-02
            1.01099369360692845e-01 -1.76794346708571387e-01 3.52704621297440027e-02
            2.99833341592391334e-01 -1.72332998032820095e-01 3.31763292623210027e-02
            -3.61089249509639657e-14 -3.49176892809388018e-01 3.35485568986260002e-02
            3.96923888187357021e-01 -0.00000000000000000e+00 3.20404318152799975e-02
            1.99479057945349092e-01 -3.44684172025722879e-01 3.25170980428999973e-02
            3.92162655837994423e-01 -3.31429495371228344e-01 2.95259345157889985e-02
            4.88095364231182993e-01 -1.63669398880865624e-01 2.91904477783679998e-02
            9.86337822477682674e-02 -5.08383790709418415e-01 3.01153795963130001e-02
            2.92640711167463374e-01 -4.95001440072169396e-01 2.81831275312489984e-02
            5.77692059949089010e-01 -0.00000000000000000e+00 2.71911334457180000e-02
            5.71141830037303455e-01 -3.10048065431951403e-01 2.48482945353980002e-02
            -6.77285711770197472e-14 -6.54942014200837974e-01 2.61526580060140006e-02
            4.76913435247620365e-01 -4.68681151924907613e-01 2.45126233562610006e-02
            1.92318751553912259e-01 -6.46054587519337531e-01 2.52572900235590002e-02
            6.58478381546617419e-01 -1.51394485555221170e-01 2.36461227308649991e-02
            3.78603731712808700e-01 -6.19540625783731480e-01 2.26693036392340008e-02
            7.36121492301041958e-01 0.00000000000000000e+00 2.09155769792820009e-02
            6.45638689901747109e-01 -4.28940778108452936e-01 1.94655413163800015e-02
            7.29774148608199447e-01 -2.80146643030677267e-01 1.88907717343300002e-02
            9.38795692461283915e-02 -7.76422514293623522e-01 2.09435456025320009e-02
            5.53723259732349371e-01 -5.74799100190879986e-01 1.87029853997510015e-02
            2.78936025714142777e-01 -7.54684141444892043e-01 1.94299539475029995e-02
            8.02943828361882161e-01 -1.34133474530735153e-01 1.68884793692209989e-02
            4.56718886526848611e-01 -7.10433537980947882e-01 1.66075029076519984e-02
            8.64329096818152953e-01 -0.00000000000000000e+00 1.38168016900120007e-02
            7.15429139988402851e-01 -5.06087876130750680e-01 1.39556214448410005e-02
            7.95923066668507539e-01 -3.68512425019292711e-01 1.35565061129429999e-02
            -9.08527948809712211e-14 -8.78555555521719045e-01 1.55186153960420008e-02
            1.80075270656212150e-01 -8.66229704786695698e-01 1.49423774701840009e-02
            8.60876867438710169e-01 -2.34551116146870270e-01 1.12640542191960007e-02
            6.25230688271456447e-01 -6.39943832677796864e-01 1.30546169697500006e-02
            3.56373167971178573e-01 -8.28402365196414858e-01 1.32757750964469998e-02
            9.09510203421620145e-01 -1.24955864851416534e-01 8.24033243736899991e-03
            5.26569572342758918e-01 -7.62561539291639279e-01 1.08205026539480009e-02
            9.37435955843908042e-01 0.00000000000000000e+00 6.65128949758999992e-03
            7.86832621491529660e-01 -5.28784646051510099e-01 9.71467820120499974e-03
            8.69734356353419891e-01 -3.81933550363015850e-01 9.45607510431199956e-03
            8.49059361353272923e-02 -9.46656701427528890e-01 9.48236088449500025e-03
            2.54475096406450163e-01 -9.20619278554513754e-01 8.87550920682099974e-03
            6.90798661492154853e-01 -6.63135022788469919e-01 8.22297010301799997e-03
            9.32243022100450514e-01 -2.34843163294000151e-01 7.22987983584699994e-03
            4.22247620920992439e-01 -8.65984440846072401e-01 7.62928540309599969e-03
            9.68911790464866307e-01 -8.46746163904975729e-02 5.12512132482500007e-03
            5.85199700715441917e-01 -7.79896016945522796e-01 5.83541588481499996e-03
            8.45946450193910016e-01 -5.13805053163552294e-01 4.54501700811999962e-03
            -1.02417093099476906e-13 -9.90383468563787006e-01 3.33676108516799998e-03
            1.42156014410077747e-01 -9.80502708979110293e-01 3.65325269843299986e-03
            9.24114104809298786e-01 -3.58379324211971495e-01 3.99244964051199969e-03
            7.44655618644480155e-01 -6.54846385619917370e-01 3.77269570831999983e-03
            3.02698811735993722e-01 -9.44521999883340624e-01 3.62961940494299987e-03
            4.67162437196598224e-01 -8.76933112040286344e-01 3.00926783198299979e-03
            9.75836936511028985e-01 -1.88621986400414415e-01 2.89213909862900010e-03
            9.94981927702018987e-01 -0.00000000000000000e+00 2.48248634432800003e-03
            6.23447271031188688e-01 -7.76981075012071920e-01 1.92903265690900008e-03
            1.01099369360692845e-01 1.76794346708571387e-01 3.52704621297440027e-02
            2.99833341592391334e-01 1.72332998032820095e-01 3.31763292623210027e-02
            -3.61089249509639657e-14 3.49176892809388018e-01 3.35485568986260002e-02
            1.99479057945349092e-01 3.44684172025722879e-01 3.25170980428999973e-02
            3.92162655837994423e-01 3.31429495371228344e-01 2.95259345157889985e-02
            4.88095364231182993e-01 1.63669398880865624e-01 2.91904477783679998e-02
            9.86337822477682674e-02 5.08383790709418415e-01 3.01153795963130001e-02
            2.92640711167463374e-01 4.95001440072169396e-01 2.81831275312489984e-02
            5.71141830037303455e-01 3.10048065431951403e-01 2.48482945353980002e-02
            -6.77285711770197472e-14 6.54942014200837974e-01 2.61526580060140006e-02
            4.76913435247620365e-01 4.68681151924907613e-01 2.45126233562610006e-02
            1.92318751553912259e-01 6.46054587519337531e-01 2.52572900235590002e-02
            6.58478381546617419e-01 1.51394485555221170e-01 2.36461227308649991e-02
            3.78603731712808700e-01 6.19540625783731480e-01 2.26693036392340008e-02
            6.45638689901747109e-01 4.28940778108452936e-01 1.94655413163800015e-02
            7.29774148608199447e-01 2.80146643030677267e-01 1.88907717343300002e-02
            9.38795692461283915e-02 7.76422514293623522e-01 2.09435456025320009e-02
            5.53723259732349371e-01 5.74799100190879986e-01 1.87029853997510015e-02
            2.78936025714142777e-01 7.54684141444892043e-01 1.94299539475029995e-02
            8.02943828361882161e-01 1.34133474530735153e-01 1.68884793692209989e-02
            4.56718886526848611e-01 7.10433537980947882e-01 1.66075029076519984e-02
            7.15429139988402851e-01 5.06087876130750680e-01 1.39556214448410005e-02
            7.95923066668507539e-01 3.68512425019292711e-01 1.35565061129429999e-02
            -9.08527948809712211e-14 8.78555555521719045e-01 1.55186153960420008e-02
            1.80075270656212150e-01 8.66229704786695698e-01 1.49423774701840009e-02
            8.60876867438710169e-01 2.34551116146870270e-01 1.12640542191960007e-02
            6.25230688271456447e-01 6.39943832677796864e-01 1.30546169697500006e-02
            3.56373167971178573e-01 8.28402365196414858e-01 1.32757750964469998e-02
            9.09510203421620145e-01 1.24955864851416534e-01 8.24033243736899991e-03
            5.26569572342758918e-01 7.62561539291639279e-01 1.08205026539480009e-02
            7.86832621491529660e-01 5.28784646051510099e-01 9.71467820120499974e-03
            8.69734356353419891e-01 3.81933550363015850e-01 9.45607510431199956e-03
            8.49059361353272923e-02 9.46656701427528890e-01 9.48236088449500025e-03
            2.54475096406450163e-01 9.20619278554513754e-01 8.87550920682099974e-03
            6.90798661492154853e-01 6.63135022788469919e-01 8.22297010301799997e-03
            9.32243022100450514e-01 2.34843163294000151e-01 7.22987983584699994e-03
            4.22247620920992439e-01 8.65984440846072401e-01 7.62928540309599969e-03
            9.68911790464866307e-01 8.46746163904975729e-02 5.12512132482500007e-03
            5.85199700715441917e-01 7.79896016945522796e-01 5.83541588481499996e-03
            8.45946450193910016e-01 5.13805053163552294e-01 4.54501700811999962e-03
            -1.02417093099476906e-13 9.90383468563787006e-01 3.33676108516799998e-03
            1.42156014410077747e-01 9.80502708979110293e-01 3.65325269843299986e-03
            9.24114104809298786e-01 3.58379324211971495e-01 3.99244964051199969e-03
            7.44655618644480155e-01 6.54846385619917370e-01 3.77269570831999983e-03
            3.02698811735993722e-01 9.44521999883340624e-01 3.62961940494299987e-03
            4.67162437196598224e-01 8.76933112040286344e-01 3.00926783198299979e-03
            9.75836936511028985e-01 1.88621986400414415e-01 2.89213909862900010e-03
            6.23447271031188688e-01 7.76981075012071920e-01 1.92903265690900008e-03
            -2.01958986148607994e-01 2.47328425945937409e-17 3.51437839711939981e-02
            -1.01099369360692873e-01 1.76794346708571387e-01 3.52704621297440027e-02
            -2.99833341592391389e-01 1.72332998032820067e-01 3.31763292623210027e-02
            -3.96923888187357021e-01 4.86091569173768685e-17 3.20404318152799975e-02
            -1.99479057945349036e-01 3.44684172025722879e-01 3.25170980428999973e-02
            -3.92162655837994423e-01 3.31429495371228344e-01 2.95259345157889985e-02
            -4.88095364231182993e-01 1.63669398880865680e-01 2.91904477783679998e-02
            -9.86337822477682119e-02 5.08383790709418415e-01 3.01153795963130001e-02
            -2.92640711167463319e-01 4.95001440072169452e-01 2.81831275312489984e-02
            -5.77692059949089010e-01 7.07468732109492792e-17 2.71911334457180000e-02
            -5.71141830037303344e-01 3.10048065431951569e-01 2.48482945353980002e-02
            -4.76913435247620199e-01 4.68681151924907780e-01 2.45126233562610006e-02
            -1.92318751553912176e-01 6.46054587519337531e-01 2.52572900235590002e-02
            -6.58478381546617419e-01 1.51394485555221281e-01 2.36461227308649991e-02
            -3.78603731712808755e-01 6.19540625783731480e-01 2.26693036392340008e-02
            -7.36121492301041958e-01 9.01488829330044070e-17 2.09155769792820009e-02
            -6.45638689901746998e-01 4.28940778108453047e-01 1.94655413163800015e-02
            -7.29774148608199447e-01 2.80146643030677489e-01 1.88907717343300002e-02
            -9.38795692461282943e-02 7.76422514293623522e-01 2.09435456025320009e-02
            -5.53723259732349260e-01 5.74799100190880097e-01 1.87029853997510015e-02
            -2.78936025714142721e-01 7.54684141444892043e-01 1.94299539475029995e-02
            -8.02943828361882161e-01 1.34133474530735070e-01 1.68884793692209989e-02
            -4.56718886526848611e-01 7.10433537980947993e-01 1.66075029076519984e-02
            -8.64329096818152953e-01 1.05849786182827371e-16 1.38168016900120007e-02
            -7.15429139988402851e-01 5.06087876130750680e-01 1.39556214448410005e-02
            -7.95923066668507539e-01 3.68512425019292711e-01 1.35565061129429999e-02
            -1.80075270656212039e-01 8.66229704786695809e-01 1.49423774701840009e-02
            -8.60876867438710169e-01 2.34551116146870270e-01 1.12640542191960007e-02
            -6.25230688271456447e-01 6.39943832677796975e-01 1.30546169697500006e-02
            -3.56373167971178462e-01 8.28402365196414858e-01 1.32757750964469998e-02
            -9.09510203421620145e-01 1.24955864851416659e-01 8.24033243736899991e-03
            -5.26569572342758696e-01 7.62561539291639501e-01 1.08205026539480009e-02
            -9.37435955843908042e-01 1.14802794272988158e-16 6.65128949758999992e-03
            -7.86832621491529660e-01 5.28784646051510099e-01 9.71467820120499974e-03
            -8.69734356353419780e-01 3.81933550363016072e-01 9.45607510431199956e-03
            -8.49059361353271674e-02 9.46656701427528890e-01 9.48236088449500025e-03
            -2.54475096406450052e-01 9.20619278554513865e-01 8.87550920682099974e-03
            -6.90798661492154964e-01 6.63135022788469919e-01 8.22297010301799997e-03
            -9.32243022100450514e-01 2.34843163294000234e-01 7.22987983584699994e-03
            -4.22247620920992106e-01 8.65984440846072512e-01 7.62928540309599969e-03
            -9.68911790464866307e-01 8.46746163904974897e-02 5.12512132482500007e-03
            -5.85199700715441695e-01 7.79896016945522907e-01 5.83541588481499996e-03
            -8.45946450193910016e-01 5.13805053163552183e-01 4.54501700811999962e-03
            -1.42156014410077636e-01 9.80502708979110293e-01 3.65325269843299986e-03
            -9.24114104809298786e-01 3.58379324211971551e-01 3.99244964051199969e-03
            -7.44655618644480155e-01 6.54846385619917259e-01 3.77269570831999983e-03
            -3.02698811735993611e-01 9.44521999883340735e-01 3.62961940494299987e-03
            -4.67162437196598113e-01 8.76933112040286344e-01 3.00926783198299979e-03
            -9.75836936511028985e-01 1.88621986400414443e-01 2.89213909862900010e-03
            -9.94981927702018987e-01 1.21850143296974065e-16 2.48248634432800003e-03
            -6.23447271031188799e-01 7.76981075012071809e-01 1.92903265690900008e-03
            -1.01099369360692901e-01 -1.76794346708571359e-01 3.52704621297440027e-02
            -2.99833341592391445e-01 -1.72332998032820012e-01 3.31763292623210027e-02
            -1.99479057945349120e-01 -3.44684172025722879e-01 3.25170980428999973e-02
            -3.92162655837994534e-01 -3.31429495371228233e-01 2.95259345157889985e-02
            -4.88095364231183049e-01 -1.63669398880865569e-01 2.91904477783679998e-02
            -9.86337822477683368e-02 -5.08383790709418415e-01 3.01153795963130001e-02
            -2.92640711167463208e-01 -4.95001440072169507e-01 2.81831275312489984e-02
            -5.71141830037303455e-01 -3.10048065431951403e-01 2.48482945353980002e-02
            -4.76913435247620365e-01 -4.68681151924907613e-01 2.45126233562610006e-02
            -1.92318751553912037e-01 -6.46054587519337531e-01 2.52572900235590002e-02
            -6.58478381546617419e-01 -1.51394485555221114e-01 2.36461227308649991e-02
            -3.78603731712808922e-01 -6.19540625783731369e-01 2.26693036392340008e-02
            -6.45638689901747109e-01 -4.28940778108452936e-01 1.94655413163800015e-02
            -7.29774148608199447e-01 -2.80146643030677267e-01 1.88907717343300002e-02
            -9.38795692461283082e-02 -7.76422514293623522e-01 2.09435456025320009e-02
            -5.53723259732349260e-01 -5.74799100190879986e-01 1.87029853997510015e-02
            -2.78936025714142721e-01 -7.54684141444892043e-01 1.94299539475029995e-02
            -8.02943828361882272e-01 -1.34133474530734903e-01 1.68884793692209989e-02
            -4.56718886526848666e-01 -7.10433537980947771e-01 1.66075029076519984e-02
            -7.15429139988402851e-01 -5.06087876130750569e-01 1.39556214448410005e-02
            -7.95923066668507651e-01 -3.68512425019292489e-01 1.35565061129429999e-02
            -1.80075270656212066e-01 -8.66229704786695698e-01 1.49423774701840009e-02
            -8.60876867438710280e-01 -2.34551116146870076e-01 1.12640542191960007e-02
            -6.25230688271456558e-01 -6.39943832677796753e-01 1.30546169697500006e-02
            -3.56373167971178462e-01 -8.28402365196414858e-01 1.32757750964469998e-02
            -9.09510203421620145e-01 -1.24955864851416437e-01 8.24033243736899991e-03
            -5.26569572342758807e-01 -7.62561539291639279e-01 1.08205026539480009e-02
            -7.86832621491529771e-01 -5.28784646051509877e-01 9.71467820120499974e-03
            -8.69734356353419891e-01 -3.81933550363015850e-01 9.45607510431199956e-03
            -8.49059361353274034e-02 -9.46656701427528890e-01 9.48236088449500025e-03
            -2.54475096406450108e-01 -9.20619278554513754e-01 8.87550920682099974e-03
            -6.90798661492155075e-01 -6.63135022788469697e-01 8.22297010301799997e-03
            -9.32243022100450625e-01 -2.34843163293999985e-01 7.22987983584699994e-03
            -4.22247620920992328e-01 -8.65984440846072401e-01 7.62928540309599969e-03
            -9.68911790464866307e-01 -8.46746163904972537e-02 5.12512132482500007e-03
            -5.85199700715441917e-01 -7.79896016945522796e-01 5.83541588481499996e-03
            -8.45946450193910127e-01 -5.13805053163551961e-01 4.54501700811999962e-03
            -1.42156014410077858e-01 -9.80502708979110182e-01 3.65325269843299986e-03
            -9.24114104809298897e-01 -3.58379324211971328e-01 3.99244964051199969e-03
            -7.44655618644480266e-01 -6.54846385619917037e-01 3.77269570831999983e-03
            -3.02698811735994056e-01 -9.44521999883340513e-01 3.62961940494299987e-03
            -4.67162437196598335e-01 -8.76933112040286233e-01 3.00926783198299979e-03
            -9.75836936511028985e-01 -1.88621986400414221e-01 2.89213909862900010e-03
            -6.23447271031189021e-01 -7.76981075012071698e-01 1.92903265690900008e-03
            ];

    case 37

        xyw=[
            0.00000000000000000e+00 -0.00000000000000000e+00 2.92705299572159974e-02
            1.82825581257615000e-01 0.00000000000000000e+00 2.86251493784610012e-02
            2.71416935397587211e-01 1.56201485948213481e-01 2.73556783843039997e-02
            3.59552739671160526e-01 -8.32479603143956555e-04 2.67346324422120009e-02
            4.42015968729190911e-01 1.51889474754960563e-01 2.48860517295190001e-02
            3.54663075936139771e-01 3.05346416072762805e-01 2.49126004835529997e-02
            5.24108747853470391e-01 -2.01790605974927397e-03 2.37301400824460017e-02
            5.15393188431053084e-01 2.97915034854656924e-01 2.19532576680810006e-02
            5.96042758875080292e-01 1.47846420989169558e-01 2.13193933396420010e-02
            4.28613255259849446e-01 4.43980396381258247e-01 2.15192031533400011e-02
            6.70288234551808193e-01 8.72724416062375919e-04 1.97963143729050009e-02
            6.55348251429181961e-01 2.91456714815503115e-01 1.78982339907040010e-02
            5.76673560437497623e-01 4.33502571865549802e-01 1.83077635665260009e-02
            7.26979953122355749e-01 1.49786707863787255e-01 1.68435929704579997e-02
            4.85925521798022730e-01 5.69179866087256570e-01 1.74395181137740017e-02
            7.91987497930007889e-01 1.54100340232413389e-02 1.53757748537860003e-02
            7.05598295615038063e-01 4.15968121216212849e-01 1.38839287373279992e-02
            7.69566498357527551e-01 2.87745738674438256e-01 1.23769427181060002e-02
            6.24363580565649445e-01 5.46666227994391574e-01 1.41840142555979994e-02
            8.29688488003787472e-01 1.67952326011657632e-01 1.20706661181599999e-02
            5.24565431472284294e-01 6.71169597290284026e-01 1.31455701796049998e-02
            8.81858730351493758e-01 3.91209417634637083e-02 1.11174619745919995e-02
            8.28531568335273860e-01 3.52201478807725232e-01 1.10537912979379992e-02
            7.54696548614217644e-01 4.94623687024500602e-01 1.13582830587450005e-02
            6.60030749453971333e-01 6.27961348383190843e-01 1.07125260799389999e-02
            8.93100037027591998e-01 2.13629505733391373e-01 9.65123650793399922e-03
            5.44666227749565857e-01 7.47389318304584549e-01 9.30993688111799930e-03
            9.40685313205889218e-01 7.11957286208994983e-02 7.84543527640699928e-03
            8.70772057060911653e-01 4.00588138191560028e-01 7.59586832974199974e-03
            7.91773823411289768e-01 5.43566462567922404e-01 7.26021425992299987e-03
            6.94789820714769801e-01 6.68623122581992524e-01 6.25144273015400005e-03
            9.31807122490403983e-01 2.55212336676506613e-01 5.86071017532800026e-03
            5.91850341327994967e-01 7.66345349116756669e-01 4.48871784126400001e-03
            9.74877740605781917e-01 -1.36514523848543096e-02 3.78715959370500010e-03
            9.71150044802023382e-01 1.35276089527840898e-01 3.71503198443900009e-03
            8.65483018995327869e-01 4.84973027925950761e-01 3.27275102072499980e-03
            9.32274734616943546e-01 3.40418647142126107e-01 3.01168444083699987e-03
            7.75389928727366784e-01 6.20106322592567372e-01 2.98273304549400006e-03
            6.67874720362860108e-01 7.35456197331994521e-01 2.50028758572299996e-03
            5.66363576327572504e-01 8.17811690186526774e-01 1.54068069911399998e-03
            9.94968238685241313e-01 3.89739338107820063e-02 1.53462167923000005e-03
            9.76903203502830553e-01 1.96802762659391561e-01 1.51135293790500007e-03
            9.14127906288075137e-02 1.58331597830750709e-01 2.86251493784610012e-02
            4.34012758762765296e-04 3.13154704045737087e-01 2.73556783843039997e-02
            1.80497318320035377e-01 3.10965566753945932e-01 2.67346324422120009e-02
            8.94678406593245323e-02 4.58741795175347633e-01 2.48860517295190001e-02
            -8.71062153054756594e-02 4.59820441581407890e-01 2.49126004835529997e-02
            2.63801931836928738e-01 4.52882536956883519e-01 2.37301400824460017e-02
            -3.05394137932729268e-04 5.95301111546080453e-01 2.19532576680810006e-02
            1.69982623002310596e-01 5.90111381422166903e-01 2.13193933396420010e-02
            -1.70191674418529448e-01 5.93180165644403012e-01 2.15192031533400011e-02
            3.34388315761091237e-01 5.80923001187719446e-01 1.97963143729050009e-02
            7.52652065808089299e-02 7.13276591471134647e-01 1.78982339907040010e-02
            -8.70874596227065312e-02 7.16165238962468642e-01 1.83077635665260009e-02
            2.33770882401899860e-01 7.04476461377874008e-01 1.68435929704579997e-02
            -2.49961462455177585e-01 7.05413779267924923e-01 1.74395181137740017e-02
            3.82648268027694471e-01 6.93586309698682957e-01 1.53757748537860003e-02
            -7.43981233020592214e-03 8.19050109477731492e-01 1.38839287373279992e-02
            1.35588129655981965e-01 8.10337006816273409e-01 1.23769427181060002e-02
            -1.61245050551334257e-01 8.14047835964860189e-01 1.41840142555979994e-02
            2.69393263051112442e-01 8.02507470844609161e-01 1.20706661181599999e-02
            -3.18967205765015027e-01 7.89871788247285433e-01 1.31455701796049998e-02
            4.07049635788615749e-01 7.83272533915216562e-01 1.11174619745919995e-02
            1.09250356269700352e-01 8.93630125419572319e-01 1.10537912979379992e-02
            -5.10084039696321601e-02 9.00898226760600385e-01 1.13582830587450005e-02
            -2.13815105567587704e-01 8.85584070497616560e-01 1.07125260799389999e-02
            2.61541439550765864e-01 8.80262073053413197e-01 9.65123650793399922e-03
            -3.74925022294121224e-01 8.45389448966857082e-01 9.30993688111799930e-03
            4.08685346976302932e-01 8.50255242513671017e-01 7.84543527640699928e-03
            8.84665244018535968e-02 9.54404791416162301e-01 7.59586832974199974e-03
            -7.48554535234190466e-02 9.57479476409672237e-01 7.26021425992299987e-03
            -2.31649699356297312e-01 9.36017196320822586e-01 6.25144273015400005e-03
            2.44883194324160391e-01 9.34574807842221356e-01 5.86071017532800026e-03
            -3.67749369743168242e-01 8.95730105386913000e-01 4.48871784126400001e-03
            4.99261374866728613e-01 8.37443162756156267e-01 3.78715959370500010e-03
            3.68422492345283514e-01 9.08678654448868528e-01 3.71503198443900009e-03
            1.27425471635305362e-02 9.92016794956979142e-01 3.27275102072499980e-03
            1.71326170961459717e-01 9.77582927055731798e-01 3.01168444083699987e-03
            -1.49332864048827979e-01 9.81560537412788703e-01 2.98273304549400006e-03
            -3.02986390078778256e-01 9.46124573045662220e-01 2.50028758572299996e-03
            -4.25063911049634646e-01 8.99391089971148250e-01 1.54068069911399998e-03
            4.63731702577070304e-01 8.81154737565468804e-01 1.53462167923000005e-03
            3.18015409753422740e-01 9.44424372601546214e-01 1.51135293790500007e-03
            -9.14127906288074582e-02 1.58331597830750737e-01 2.86251493784610012e-02
            -2.70982922638824453e-01 1.56953218097523661e-01 2.73556783843039997e-02
            -1.79055421351125121e-01 3.11798046357090020e-01 2.67346324422120009e-02
            -3.52548128069866407e-01 3.06852320420387237e-01 2.48860517295190001e-02
            -4.41769291241615458e-01 1.54474025508645196e-01 2.49126004835529997e-02
            -2.60306816016541598e-01 4.54900443016632905e-01 2.37301400824460017e-02
            -5.15698582568985819e-01 2.97386076691423695e-01 2.19532576680810006e-02
            -4.26060135872769752e-01 4.42264960432997400e-01 2.13193933396420010e-02
            -5.98804929678379061e-01 1.49199769263144710e-01 2.15192031533400011e-02
            -3.35899918790716956e-01 5.80050276771657103e-01 1.97963143729050009e-02
            -5.80083044848372920e-01 4.21819876655631643e-01 1.78982339907040010e-02
            -6.63761020060204099e-01 2.82662667096919007e-01 1.83077635665260009e-02
            -4.93209070720455889e-01 5.54689753514086892e-01 1.68435929704579997e-02
            -7.35886984253200427e-01 1.36233913180668409e-01 1.74395181137740017e-02
            -4.09339229902313306e-01 6.78176275675441809e-01 1.53757748537860003e-02
            -7.13038107945244026e-01 4.03081988261518809e-01 1.38839287373279992e-02
            -6.33978368701545669e-01 5.22591268141835208e-01 1.23769427181060002e-02
            -7.85608631116983536e-01 2.67381607970468893e-01 1.41840142555979994e-02
            -5.60295224952675142e-01 6.34555144832951723e-01 1.20706661181599999e-02
            -8.43532637237299432e-01 1.18702190957001685e-01 1.31455701796049998e-02
            -4.74809094562877843e-01 7.44151592151753083e-01 1.11174619745919995e-02
            -7.19281212065573494e-01 5.41428646611847197e-01 1.10537912979379992e-02
            -8.05704952583849776e-01 4.06274539736100004e-01 1.13582830587450005e-02
            -8.73845855021558982e-01 2.57622722114426050e-01 1.07125260799389999e-02
            -6.31558597476826078e-01 6.66632567320022185e-01 9.65123650793399922e-03
            -9.19591250043687136e-01 9.80001306622728385e-02 9.30993688111799930e-03
            -5.31999966229586341e-01 7.79059513892771505e-01 7.84543527640699928e-03
            -7.82305532659058001e-01 5.53816653224602384e-01 7.59586832974199974e-03
            -8.66629276934708703e-01 4.13913013841750110e-01 7.26021425992299987e-03
            -9.26439520071067113e-01 2.67394073738830396e-01 6.25144273015400005e-03
            -6.86923928166243702e-01 6.79362471165714910e-01 5.86071017532800026e-03
            -9.59599711071163264e-01 1.29384756270156553e-01 4.48871784126400001e-03
            -4.75616365739053415e-01 8.51094615141010813e-01 3.78715959370500010e-03
            -6.02727552456739923e-01 7.73402564921027769e-01 3.71503198443900009e-03
            -8.52740471831797131e-01 5.07043767031028603e-01 3.27275102072499980e-03
            -7.60948563655483690e-01 6.37164279913605913e-01 3.01168444083699987e-03
            -9.24722792776194735e-01 3.61454214820221720e-01 2.98273304549400006e-03
            -9.70861110441638253e-01 2.10668375713668060e-01 2.50028758572299996e-03
            -9.91427487377207317e-01 8.15793997846214480e-02 1.54068069911399998e-03
            -5.31236536108171009e-01 8.42180803754686957e-01 1.53462167923000005e-03
            -6.58887793749407535e-01 7.47621609942154985e-01 1.51135293790500007e-03
            -1.82825581257615000e-01 2.23896762889392544e-17 2.86251493784610012e-02
            -2.71416935397587211e-01 -1.56201485948213481e-01 2.73556783843039997e-02
            -3.59552739671160526e-01 8.32479603144023993e-04 2.67346324422120009e-02
            -4.42015968729190967e-01 -1.51889474754960480e-01 2.48860517295190001e-02
            -3.54663075936139827e-01 -3.05346416072762750e-01 2.49126004835529997e-02
            -5.24108747853470391e-01 2.01790605974937198e-03 2.37301400824460017e-02
            -5.15393188431053195e-01 -2.97915034854656813e-01 2.19532576680810006e-02
            -5.96042758875080292e-01 -1.47846420989169558e-01 2.13193933396420010e-02
            -4.28613255259849446e-01 -4.43980396381258247e-01 2.15192031533400011e-02
            -6.70288234551808193e-01 -8.72724416062271510e-04 1.97963143729050009e-02
            -6.55348251429181961e-01 -2.91456714815503060e-01 1.78982339907040010e-02
            -5.76673560437497623e-01 -4.33502571865549802e-01 1.83077635665260009e-02
            -7.26979953122355749e-01 -1.49786707863787144e-01 1.68435929704579997e-02
            -4.85925521798022897e-01 -5.69179866087256348e-01 1.74395181137740017e-02
            -7.91987497930007889e-01 -1.54100340232412401e-02 1.53757748537860003e-02
            -7.05598295615038063e-01 -4.15968121216212794e-01 1.38839287373279992e-02
            -7.69566498357527551e-01 -2.87745738674438256e-01 1.23769427181060002e-02
            -6.24363580565649556e-01 -5.46666227994391463e-01 1.41840142555979994e-02
            -8.29688488003787472e-01 -1.67952326011657632e-01 1.20706661181599999e-02
            -5.24565431472284183e-01 -6.71169597290284026e-01 1.31455701796049998e-02
            -8.81858730351493758e-01 -3.91209417634635973e-02 1.11174619745919995e-02
            -8.28531568335273860e-01 -3.52201478807725177e-01 1.10537912979379992e-02
            -7.54696548614217755e-01 -4.94623687024500547e-01 1.13582830587450005e-02
            -6.60030749453971555e-01 -6.27961348383190621e-01 1.07125260799389999e-02
            -8.93100037027591998e-01 -2.13629505733391428e-01 9.65123650793399922e-03
            -5.44666227749566079e-01 -7.47389318304584438e-01 9.30993688111799930e-03
            -9.40685313205889218e-01 -7.11957286208995815e-02 7.84543527640699928e-03
            -8.70772057060911653e-01 -4.00588138191559973e-01 7.59586832974199974e-03
            -7.91773823411289879e-01 -5.43566462567922182e-01 7.26021425992299987e-03
            -6.94789820714769690e-01 -6.68623122581992635e-01 6.25144273015400005e-03
            -9.31807122490404094e-01 -2.55212336676506502e-01 5.86071017532800026e-03
            -5.91850341327994744e-01 -7.66345349116756891e-01 4.48871784126400001e-03
            -9.74877740605781917e-01 1.36514523848542749e-02 3.78715959370500010e-03
            -9.71150044802023382e-01 -1.35276089527840843e-01 3.71503198443900009e-03
            -8.65483018995327980e-01 -4.84973027925950595e-01 3.27275102072499980e-03
            -9.32274734616943546e-01 -3.40418647142126052e-01 3.01168444083699987e-03
            -7.75389928727366784e-01 -6.20106322592567483e-01 2.98273304549400006e-03
            -6.67874720362860219e-01 -7.35456197331994299e-01 2.50028758572299996e-03
            -5.66363576327572504e-01 -8.17811690186526774e-01 1.54068069911399998e-03
            -9.94968238685241313e-01 -3.89739338107818883e-02 1.53462167923000005e-03
            -9.76903203502830553e-01 -1.96802762659391645e-01 1.51135293790500007e-03
            -9.14127906288075831e-02 -1.58331597830750709e-01 2.86251493784610012e-02
            -4.34012758762942780e-04 -3.13154704045737087e-01 2.73556783843039997e-02
            -1.80497318320035460e-01 -3.10965566753945877e-01 2.67346324422120009e-02
            -8.94678406593246850e-02 -4.58741795175347633e-01 2.48860517295190001e-02
            8.71062153054757010e-02 -4.59820441581407890e-01 2.49126004835529997e-02
            -2.63801931836928683e-01 -4.52882536956883575e-01 2.37301400824460017e-02
            3.05394137932788520e-04 -5.95301111546080453e-01 2.19532576680810006e-02
            -1.69982623002310901e-01 -5.90111381422166792e-01 2.13193933396420010e-02
            1.70191674418529143e-01 -5.93180165644403012e-01 2.15192031533400011e-02
            -3.34388315761091404e-01 -5.80923001187719223e-01 1.97963143729050009e-02
            -7.52652065808091797e-02 -7.13276591471134647e-01 1.78982339907040010e-02
            8.70874596227059622e-02 -7.16165238962468753e-01 1.83077635665260009e-02
            -2.33770882401899832e-01 -7.04476461377874008e-01 1.68435929704579997e-02
            2.49961462455177169e-01 -7.05413779267925145e-01 1.74395181137740017e-02
            -3.82648268027694749e-01 -6.93586309698682846e-01 1.53757748537860003e-02
            7.43981233020563938e-03 -8.19050109477731492e-01 1.38839287373279992e-02
            -1.35588129655982437e-01 -8.10337006816273298e-01 1.23769427181060002e-02
            1.61245050551333813e-01 -8.14047835964860300e-01 1.41840142555979994e-02
            -2.69393263051112886e-01 -8.02507470844609050e-01 1.20706661181599999e-02
            3.18967205765014972e-01 -7.89871788247285433e-01 1.31455701796049998e-02
            -4.07049635788616082e-01 -7.83272533915216340e-01 1.11174619745919995e-02
            -1.09250356269700657e-01 -8.93630125419572319e-01 1.10537912979379992e-02
            5.10084039696320560e-02 -9.00898226760600385e-01 1.13582830587450005e-02
            2.13815105567587593e-01 -8.85584070497616560e-01 1.07125260799389999e-02
            -2.61541439550765920e-01 -8.80262073053413086e-01 9.65123650793399922e-03
            3.74925022294120780e-01 -8.45389448966857304e-01 9.30993688111799930e-03
            -4.08685346976303376e-01 -8.50255242513670795e-01 7.84543527640699928e-03
            -8.84665244018539298e-02 -9.54404791416162301e-01 7.59586832974199974e-03
            7.48554535234189355e-02 -9.57479476409672237e-01 7.26021425992299987e-03
            2.31649699356297201e-01 -9.36017196320822586e-01 6.25144273015400005e-03
            -2.44883194324160336e-01 -9.34574807842221356e-01 5.86071017532800026e-03
            3.67749369743167520e-01 -8.95730105386913333e-01 4.48871784126400001e-03
            -4.99261374866729057e-01 -8.37443162756156045e-01 3.78715959370500010e-03
            -3.68422492345284236e-01 -9.08678654448868195e-01 3.71503198443900009e-03
            -1.27425471635306577e-02 -9.92016794956979142e-01 3.27275102072499980e-03
            -1.71326170961460050e-01 -9.77582927055731798e-01 3.01168444083699987e-03
            1.49332864048827840e-01 -9.81560537412788703e-01 2.98273304549400006e-03
            3.02986390078777701e-01 -9.46124573045662332e-01 2.50028758572299996e-03
            4.25063911049634535e-01 -8.99391089971148250e-01 1.54068069911399998e-03
            -4.63731702577070970e-01 -8.81154737565468471e-01 1.53462167923000005e-03
            -3.18015409753422906e-01 -9.44424372601546214e-01 1.51135293790500007e-03
            9.14127906288073749e-02 -1.58331597830750792e-01 2.86251493784610012e-02
            2.70982922638824286e-01 -1.56953218097523883e-01 2.73556783843039997e-02
            1.79055421351124955e-01 -3.11798046357090075e-01 2.67346324422120009e-02
            3.52548128069866240e-01 -3.06852320420387459e-01 2.48860517295190001e-02
            4.41769291241615347e-01 -1.54474025508645640e-01 2.49126004835529997e-02
            2.60306816016541542e-01 -4.54900443016632960e-01 2.37301400824460017e-02
            5.15698582568985486e-01 -2.97386076691424250e-01 2.19532576680810006e-02
            4.26060135872769308e-01 -4.42264960432997845e-01 2.13193933396420010e-02
            5.98804929678378839e-01 -1.49199769263145321e-01 2.15192031533400011e-02
            3.35899918790716567e-01 -5.80050276771657325e-01 1.97963143729050009e-02
            5.80083044848372698e-01 -4.21819876655631976e-01 1.78982339907040010e-02
            6.63761020060204099e-01 -2.82662667096919062e-01 1.83077635665260009e-02
            4.93209070720455778e-01 -5.54689753514087003e-01 1.68435929704579997e-02
            7.35886984253200205e-01 -1.36233913180669158e-01 1.74395181137740017e-02
            4.09339229902312918e-01 -6.78176275675441920e-01 1.53757748537860003e-02
            7.13038107945243693e-01 -4.03081988261519197e-01 1.38839287373279992e-02
            6.33978368701545114e-01 -5.22591268141835874e-01 1.23769427181060002e-02
            7.85608631116983536e-01 -2.67381607970469337e-01 1.41840142555979994e-02
            5.60295224952674475e-01 -6.34555144832952278e-01 1.20706661181599999e-02
            8.43532637237299432e-01 -1.18702190957001796e-01 1.31455701796049998e-02
            4.74809094562877454e-01 -7.44151592151753194e-01 1.11174619745919995e-02
            7.19281212065573161e-01 -5.41428646611847753e-01 1.10537912979379992e-02
            8.05704952583849665e-01 -4.06274539736100115e-01 1.13582830587450005e-02
            8.73845855021558870e-01 -2.57622722114426161e-01 1.07125260799389999e-02
            6.31558597476825967e-01 -6.66632567320022185e-01 9.65123650793399922e-03
            9.19591250043687025e-01 -9.80001306622733659e-02 9.30993688111799930e-03
            5.31999966229585564e-01 -7.79059513892772171e-01 7.84543527640699928e-03
            7.82305532659057667e-01 -5.53816653224602828e-01 7.59586832974199974e-03
            8.66629276934708703e-01 -4.13913013841750221e-01 7.26021425992299987e-03
            9.26439520071067113e-01 -2.67394073738830507e-01 6.25144273015400005e-03
            6.86923928166243591e-01 -6.79362471165714910e-01 5.86071017532800026e-03
            9.59599711071163264e-01 -1.29384756270156664e-01 4.48871784126400001e-03
            4.75616365739052582e-01 -8.51094615141011257e-01 3.78715959370500010e-03
            6.02727552456739146e-01 -7.73402564921028213e-01 3.71503198443900009e-03
            8.52740471831797020e-01 -5.07043767031028714e-01 3.27275102072499980e-03
            7.60948563655483357e-01 -6.37164279913606357e-01 3.01168444083699987e-03
            9.24722792776194735e-01 -3.61454214820221831e-01 2.98273304549400006e-03
            9.70861110441638142e-01 -2.10668375713668615e-01 2.50028758572299996e-03
            9.91427487377207206e-01 -8.15793997846220170e-02 1.54068069911399998e-03
            5.31236536108170121e-01 -8.42180803754687513e-01 1.53462167923000005e-03
            6.58887793749407424e-01 -7.47621609942155096e-01 1.51135293790500007e-03
            ];

    case 39

        xyw=[
            0.00000000000000000e+00 -0.00000000000000000e+00 2.45100865534200012e-02
            8.83869469791783019e-02 -1.37495641599950202e-01 2.41069828235210004e-02
            1.76591591789478003e-01 -0.00000000000000000e+00 2.39874089922159986e-02
            -2.82256175776697271e-14 -2.72944527060300979e-01 2.33317004326199992e-02
            2.61481255888243558e-01 -1.37400425631441198e-01 2.31315662859799993e-02
            1.73271809459434989e-01 -2.73207500751926180e-01 2.29201506406810016e-02
            3.45948751367827989e-01 -0.00000000000000000e+00 2.24515455092339986e-02
            8.45512423949050518e-02 -4.04319227836092920e-01 2.16247459172369989e-02
            3.40317340941899360e-01 -2.71887639083457089e-01 2.17115304575720008e-02
            4.25111714688867881e-01 -1.34983206791177701e-01 2.12476271763849985e-02
            2.50901885159939608e-01 -4.04816516392946801e-01 2.10934553054530002e-02
            5.03754206775718028e-01 -0.00000000000000000e+00 1.99598280653919997e-02
            -5.46016242753229636e-14 -5.28003133095047983e-01 1.90825783920320000e-02
            1.59451769149947575e-01 -5.29765864797819064e-01 1.91793276364020000e-02
            4.97734521996865131e-01 -2.62657009518657836e-01 1.96745374638920001e-02
            4.11476369725435998e-01 -3.98592214858742966e-01 1.97203484545829984e-02
            5.77620953356464795e-01 -1.25465112849677152e-01 1.87191085309930012e-02
            3.18413825446693277e-01 -5.26744975747317379e-01 1.86046575855860000e-02
            6.94711365712900736e-02 -6.42855223486602290e-01 1.52141447952580007e-02
            6.52739984406775053e-01 -0.00000000000000000e+00 1.71395927008720014e-02
            2.17485065236955571e-01 -6.42862063898548608e-01 1.61202993243050006e-02
            5.52383208993478148e-01 -3.95289818119472947e-01 1.72680306896660016e-02
            6.31654473642279113e-01 -2.60327056202154294e-01 1.62660549637680014e-02
            4.62232590269700094e-01 -5.26420749996411308e-01 1.70375192273710015e-02
            7.07081716647408265e-01 -1.38709569152842327e-01 1.54291555944510005e-02
            3.57344408660123181e-01 -6.47055757185271974e-01 1.56894836518619990e-02
            8.80054955548493362e-02 -7.37524290774686286e-01 1.23839595503690003e-02
            7.62317928585678040e-01 -0.00000000000000000e+00 1.28757066367399999e-02
            -8.03127847092832719e-14 -7.76632609686936037e-01 7.20646815938200038e-03
            6.85782725509727964e-01 -3.65372977667582644e-01 1.41650720215729994e-02
            6.02958838327382285e-01 -5.02139307676215063e-01 1.48370881953289995e-02
            2.25347624992565776e-01 -7.52793309411134981e-01 1.47614709123309994e-02
            7.63888747257869705e-01 -2.45084093906073247e-01 1.31654615542190004e-02
            5.01406983635098169e-01 -6.30278174345496711e-01 1.42269552899920004e-02
            8.14590318312071182e-01 -1.02414667496028358e-01 1.12948370918710007e-02
            3.76577165676810488e-01 -7.44258193563618997e-01 1.32624073080989998e-02
            8.97075099105037743e-02 -8.38740331484917556e-01 1.32318606149549999e-02
            7.28692435382443326e-01 -4.56246819427198580e-01 1.19587045362549993e-02
            6.35145681044101740e-01 -5.95337224218106642e-01 1.19101599437899993e-02
            2.46134915729779441e-01 -8.38552835449991707e-01 1.15687313960700002e-02
            8.10613049334679370e-01 -3.31195940831167435e-01 1.08246537112860006e-02
            8.75794443330869976e-01 -0.00000000000000000e+00 1.08463365309339997e-02
            8.64329102915658232e-01 -1.82703866281846233e-01 9.92974474843799984e-03
            5.18299056778332545e-01 -7.19194708013316308e-01 1.11675565859830003e-02
            -9.38281878131612361e-14 -9.07327900872903048e-01 9.47529405165799948e-03
            3.92557793518652498e-01 -8.21643113621251198e-01 9.65808471410700004e-03
            1.47148525709282968e-01 -9.10349762177830701e-01 8.54249027516700010e-03
            7.48630999239305428e-01 -5.41904601479935910e-01 8.83271956823299988e-03
            9.27028017121971137e-01 -8.79632471159870616e-02 8.45223553558799995e-03
            8.35618919688791317e-01 -4.15684930783213058e-01 8.20610183142100012e-03
            6.41552974427336831e-01 -6.79161816231979132e-01 8.53654735225200037e-03
            8.98182944079939705e-01 -2.64756390895498295e-01 7.82744669730900053e-03
            2.87250883888211184e-01 -8.99029216084602290e-01 7.04244263164499965e-03
            5.21594843533790486e-01 -7.91992590510683381e-01 7.48864467749500023e-03
            6.67644208727448257e-02 -9.60719825750044998e-01 5.59393254402100037e-03
            4.10164775379314950e-01 -8.76790329774203903e-01 5.31464774268599997e-03
            7.45695427994018001e-01 -6.17220828768574203e-01 5.60859577042399984e-03
            9.71209550667442967e-01 -0.00000000000000000e+00 5.61697330160400039e-03
            9.56212171764805996e-01 -1.73733787837638487e-01 5.49557138464999966e-03
            8.39277183103726898e-01 -4.93333155706183413e-01 5.12874570092299985e-03
            9.11513026544148453e-01 -3.44951464831909471e-01 5.04166234001500024e-03
            2.09116655722167338e-01 -9.52674216498498372e-01 4.80417333523300034e-03
            6.33554827683826205e-01 -7.45131826075071113e-01 4.92955357600600042e-03
            5.19523983143803547e-01 -8.41268061702491909e-01 3.48602570293200017e-03
            3.39444264710094923e-01 -9.29721150164554144e-01 3.00351288049500011e-03
            9.19944100697847322e-02 -9.88328121673905269e-01 2.24411252693399984e-03
            7.32575755985332089e-01 -6.71136450425255160e-01 2.39669624140899980e-03
            -1.02758177492695917e-13 -9.93681788543452016e-01 1.18845468766399999e-03
            9.90823718512418350e-01 -8.54860139910965239e-02 2.40780314138600016e-03
            9.61810650834861836e-01 -2.53784640031552100e-01 2.30224558693400017e-03
            8.25422851388604384e-01 -5.55542598048378244e-01 2.02948103005200003e-03
            9.04898777443704949e-01 -4.14932055916150955e-01 1.98388331462500019e-03
            2.23305163996555395e-01 -9.72537036373102626e-01 1.15696656985899999e-03
            6.26164145956925711e-01 -7.78678481791463706e-01 9.55709298755000021e-04
            4.36004265640657196e-01 -8.99317902617156939e-01 7.72276072839000040e-04
            8.83869469791783019e-02 1.37495641599950202e-01 2.41069828235210004e-02
            -2.82256175776697271e-14 2.72944527060300979e-01 2.33317004326199992e-02
            2.61481255888243558e-01 1.37400425631441198e-01 2.31315662859799993e-02
            1.73271809459434989e-01 2.73207500751926180e-01 2.29201506406810016e-02
            8.45512423949050518e-02 4.04319227836092920e-01 2.16247459172369989e-02
            3.40317340941899360e-01 2.71887639083457089e-01 2.17115304575720008e-02
            4.25111714688867881e-01 1.34983206791177701e-01 2.12476271763849985e-02
            2.50901885159939608e-01 4.04816516392946801e-01 2.10934553054530002e-02
            -5.46016242753229636e-14 5.28003133095047983e-01 1.90825783920320000e-02
            1.59451769149947575e-01 5.29765864797819064e-01 1.91793276364020000e-02
            4.97734521996865131e-01 2.62657009518657836e-01 1.96745374638920001e-02
            4.11476369725435998e-01 3.98592214858742966e-01 1.97203484545829984e-02
            5.77620953356464795e-01 1.25465112849677152e-01 1.87191085309930012e-02
            3.18413825446693277e-01 5.26744975747317379e-01 1.86046575855860000e-02
            6.94711365712900736e-02 6.42855223486602290e-01 1.52141447952580007e-02
            2.17485065236955571e-01 6.42862063898548608e-01 1.61202993243050006e-02
            5.52383208993478148e-01 3.95289818119472947e-01 1.72680306896660016e-02
            6.31654473642279113e-01 2.60327056202154294e-01 1.62660549637680014e-02
            4.62232590269700094e-01 5.26420749996411308e-01 1.70375192273710015e-02
            7.07081716647408265e-01 1.38709569152842327e-01 1.54291555944510005e-02
            3.57344408660123181e-01 6.47055757185271974e-01 1.56894836518619990e-02
            8.80054955548493362e-02 7.37524290774686286e-01 1.23839595503690003e-02
            -8.03127847092832719e-14 7.76632609686936037e-01 7.20646815938200038e-03
            6.85782725509727964e-01 3.65372977667582644e-01 1.41650720215729994e-02
            6.02958838327382285e-01 5.02139307676215063e-01 1.48370881953289995e-02
            2.25347624992565776e-01 7.52793309411134981e-01 1.47614709123309994e-02
            7.63888747257869705e-01 2.45084093906073247e-01 1.31654615542190004e-02
            5.01406983635098169e-01 6.30278174345496711e-01 1.42269552899920004e-02
            8.14590318312071182e-01 1.02414667496028358e-01 1.12948370918710007e-02
            3.76577165676810488e-01 7.44258193563618997e-01 1.32624073080989998e-02
            8.97075099105037743e-02 8.38740331484917556e-01 1.32318606149549999e-02
            7.28692435382443326e-01 4.56246819427198580e-01 1.19587045362549993e-02
            6.35145681044101740e-01 5.95337224218106642e-01 1.19101599437899993e-02
            2.46134915729779441e-01 8.38552835449991707e-01 1.15687313960700002e-02
            8.10613049334679370e-01 3.31195940831167435e-01 1.08246537112860006e-02
            8.64329102915658232e-01 1.82703866281846233e-01 9.92974474843799984e-03
            5.18299056778332545e-01 7.19194708013316308e-01 1.11675565859830003e-02
            -9.38281878131612361e-14 9.07327900872903048e-01 9.47529405165799948e-03
            3.92557793518652498e-01 8.21643113621251198e-01 9.65808471410700004e-03
            1.47148525709282968e-01 9.10349762177830701e-01 8.54249027516700010e-03
            7.48630999239305428e-01 5.41904601479935910e-01 8.83271956823299988e-03
            9.27028017121971137e-01 8.79632471159870616e-02 8.45223553558799995e-03
            8.35618919688791317e-01 4.15684930783213058e-01 8.20610183142100012e-03
            6.41552974427336831e-01 6.79161816231979132e-01 8.53654735225200037e-03
            8.98182944079939705e-01 2.64756390895498295e-01 7.82744669730900053e-03
            2.87250883888211184e-01 8.99029216084602290e-01 7.04244263164499965e-03
            5.21594843533790486e-01 7.91992590510683381e-01 7.48864467749500023e-03
            6.67644208727448257e-02 9.60719825750044998e-01 5.59393254402100037e-03
            4.10164775379314950e-01 8.76790329774203903e-01 5.31464774268599997e-03
            7.45695427994018001e-01 6.17220828768574203e-01 5.60859577042399984e-03
            9.56212171764805996e-01 1.73733787837638487e-01 5.49557138464999966e-03
            8.39277183103726898e-01 4.93333155706183413e-01 5.12874570092299985e-03
            9.11513026544148453e-01 3.44951464831909471e-01 5.04166234001500024e-03
            2.09116655722167338e-01 9.52674216498498372e-01 4.80417333523300034e-03
            6.33554827683826205e-01 7.45131826075071113e-01 4.92955357600600042e-03
            5.19523983143803547e-01 8.41268061702491909e-01 3.48602570293200017e-03
            3.39444264710094923e-01 9.29721150164554144e-01 3.00351288049500011e-03
            9.19944100697847322e-02 9.88328121673905269e-01 2.24411252693399984e-03
            7.32575755985332089e-01 6.71136450425255160e-01 2.39669624140899980e-03
            -1.02758177492695917e-13 9.93681788543452016e-01 1.18845468766399999e-03
            9.90823718512418350e-01 8.54860139910965239e-02 2.40780314138600016e-03
            9.61810650834861836e-01 2.53784640031552100e-01 2.30224558693400017e-03
            8.25422851388604384e-01 5.55542598048378244e-01 2.02948103005200003e-03
            9.04898777443704949e-01 4.14932055916150955e-01 1.98388331462500019e-03
            2.23305163996555395e-01 9.72537036373102626e-01 1.15696656985899999e-03
            6.26164145956925711e-01 7.78678481791463706e-01 9.55709298755000021e-04
            4.36004265640657196e-01 8.99317902617156939e-01 7.72276072839000040e-04
            -8.83869469791782880e-02 1.37495641599950230e-01 2.41069828235210004e-02
            -1.76591591789478003e-01 2.16262327641320259e-17 2.39874089922159986e-02
            -2.61481255888243558e-01 1.37400425631441198e-01 2.31315662859799993e-02
            -1.73271809459434933e-01 2.73207500751926236e-01 2.29201506406810016e-02
            -3.45948751367827989e-01 4.23665031031634054e-17 2.24515455092339986e-02
            -8.45512423949050101e-02 4.04319227836092920e-01 2.16247459172369989e-02
            -3.40317340941899360e-01 2.71887639083457089e-01 2.17115304575720008e-02
            -4.25111714688867881e-01 1.34983206791177729e-01 2.12476271763849985e-02
            -2.50901885159939442e-01 4.04816516392946857e-01 2.10934553054530002e-02
            -5.03754206775718028e-01 6.16920976884896995e-17 1.99598280653919997e-02
            -1.59451769149947492e-01 5.29765864797819064e-01 1.91793276364020000e-02
            -4.97734521996865131e-01 2.62657009518657836e-01 1.96745374638920001e-02
            -4.11476369725435887e-01 3.98592214858743077e-01 1.97203484545829984e-02
            -5.77620953356464795e-01 1.25465112849677179e-01 1.87191085309930012e-02
            -3.18413825446693055e-01 5.26744975747317601e-01 1.86046575855860000e-02
            -6.94711365712899903e-02 6.42855223486602290e-01 1.52141447952580007e-02
            -6.52739984406775053e-01 7.99375932579250349e-17 1.71395927008720014e-02
            -2.17485065236955516e-01 6.42862063898548608e-01 1.61202993243050006e-02
            -5.52383208993478148e-01 3.95289818119472947e-01 1.72680306896660016e-02
            -6.31654473642279002e-01 2.60327056202154461e-01 1.62660549637680014e-02
            -4.62232590269699872e-01 5.26420749996411419e-01 1.70375192273710015e-02
            -7.07081716647408265e-01 1.38709569152842410e-01 1.54291555944510005e-02
            -3.57344408660123014e-01 6.47055757185272196e-01 1.56894836518619990e-02
            -8.80054955548492529e-02 7.37524290774686286e-01 1.23839595503690003e-02
            -7.62317928585678040e-01 9.33570211175091187e-17 1.28757066367399999e-02
            -6.85782725509727964e-01 3.65372977667582755e-01 1.41650720215729994e-02
            -6.02958838327382285e-01 5.02139307676215063e-01 1.48370881953289995e-02
            -2.25347624992565693e-01 7.52793309411135092e-01 1.47614709123309994e-02
            -7.63888747257869705e-01 2.45084093906073303e-01 1.31654615542190004e-02
            -5.01406983635098169e-01 6.30278174345496711e-01 1.42269552899920004e-02
            -8.14590318312071071e-01 1.02414667496028552e-01 1.12948370918710007e-02
            -3.76577165676810599e-01 7.44258193563618886e-01 1.32624073080989998e-02
            -8.97075099105036911e-02 8.38740331484917556e-01 1.32318606149549999e-02
            -7.28692435382443215e-01 4.56246819427198691e-01 1.19587045362549993e-02
            -6.35145681044101629e-01 5.95337224218106642e-01 1.19101599437899993e-02
            -2.46134915729779330e-01 8.38552835449991818e-01 1.15687313960700002e-02
            -8.10613049334679370e-01 3.31195940831167435e-01 1.08246537112860006e-02
            -8.75794443330869976e-01 1.07253886173618797e-16 1.08463365309339997e-02
            -8.64329102915658232e-01 1.82703866281846178e-01 9.92974474843799984e-03
            -5.18299056778332323e-01 7.19194708013316419e-01 1.11675565859830003e-02
            -3.92557793518652387e-01 8.21643113621251309e-01 9.65808471410700004e-03
            -1.47148525709282857e-01 9.10349762177830701e-01 8.54249027516700010e-03
            -7.48630999239305428e-01 5.41904601479935910e-01 8.83271956823299988e-03
            -9.27028017121971137e-01 8.79632471159870755e-02 8.45223553558799995e-03
            -8.35618919688791317e-01 4.15684930783213114e-01 8.20610183142100012e-03
            -6.41552974427336609e-01 6.79161816231979243e-01 8.53654735225200037e-03
            -8.98182944079939816e-01 2.64756390895498295e-01 7.82744669730900053e-03
            -2.87250883888211017e-01 8.99029216084602290e-01 7.04244263164499965e-03
            -5.21594843533790486e-01 7.91992590510683381e-01 7.48864467749500023e-03
            -6.67644208727447008e-02 9.60719825750045109e-01 5.59393254402100037e-03
            -4.10164775379315005e-01 8.76790329774203792e-01 5.31464774268599997e-03
            -7.45695427994017890e-01 6.17220828768574425e-01 5.60859577042399984e-03
            -9.71209550667442967e-01 1.18938866752622329e-16 5.61697330160400039e-03
            -9.56212171764805996e-01 1.73733787837638681e-01 5.49557138464999966e-03
            -8.39277183103726676e-01 4.93333155706183635e-01 5.12874570092299985e-03
            -9.11513026544148453e-01 3.44951464831909416e-01 5.04166234001500024e-03
            -2.09116655722167200e-01 9.52674216498498483e-01 4.80417333523300034e-03
            -6.33554827683826316e-01 7.45131826075071002e-01 4.92955357600600042e-03
            -5.19523983143803658e-01 8.41268061702491909e-01 3.48602570293200017e-03
            -3.39444264710094812e-01 9.29721150164554255e-01 3.00351288049500011e-03
            -9.19944100697846073e-02 9.88328121673905269e-01 2.24411252693399984e-03
            -7.32575755985331867e-01 6.71136450425255382e-01 2.39669624140899980e-03
            -9.90823718512418350e-01 8.54860139910964822e-02 2.40780314138600016e-03
            -9.61810650834861725e-01 2.53784640031552433e-01 2.30224558693400017e-03
            -8.25422851388604273e-01 5.55542598048378355e-01 2.02948103005200003e-03
            -9.04898777443704949e-01 4.14932055916150955e-01 1.98388331462500019e-03
            -2.23305163996555284e-01 9.72537036373102626e-01 1.15696656985899999e-03
            -6.26164145956925600e-01 7.78678481791463817e-01 9.55709298755000021e-04
            -4.36004265640657085e-01 8.99317902617156939e-01 7.72276072839000040e-04
            -8.83869469791783019e-02 -1.37495641599950202e-01 2.41069828235210004e-02
            -2.61481255888243613e-01 -1.37400425631441142e-01 2.31315662859799993e-02
            -1.73271809459435017e-01 -2.73207500751926180e-01 2.29201506406810016e-02
            -8.45512423949051906e-02 -4.04319227836092865e-01 2.16247459172369989e-02
            -3.40317340941899471e-01 -2.71887639083456978e-01 2.17115304575720008e-02
            -4.25111714688867937e-01 -1.34983206791177618e-01 2.12476271763849985e-02
            -2.50901885159939553e-01 -4.04816516392946801e-01 2.10934553054530002e-02
            -1.59451769149947742e-01 -5.29765864797819064e-01 1.91793276364020000e-02
            -4.97734521996865242e-01 -2.62657009518657725e-01 1.96745374638920001e-02
            -4.11476369725435998e-01 -3.98592214858743021e-01 1.97203484545829984e-02
            -5.77620953356464906e-01 -1.25465112849677068e-01 1.87191085309930012e-02
            -3.18413825446693222e-01 -5.26744975747317490e-01 1.86046575855860000e-02
            -6.94711365712902956e-02 -6.42855223486602179e-01 1.52141447952580007e-02
            -2.17485065236955655e-01 -6.42862063898548608e-01 1.61202993243050006e-02
            -5.52383208993478259e-01 -3.95289818119472780e-01 1.72680306896660016e-02
            -6.31654473642279113e-01 -2.60327056202154294e-01 1.62660549637680014e-02
            -4.62232590269700039e-01 -5.26420749996411308e-01 1.70375192273710015e-02
            -7.07081716647408265e-01 -1.38709569152842244e-01 1.54291555944510005e-02
            -3.57344408660123181e-01 -6.47055757185272085e-01 1.56894836518619990e-02
            -8.80054955548495860e-02 -7.37524290774686286e-01 1.23839595503690003e-02
            -6.85782725509727964e-01 -3.65372977667582588e-01 1.41650720215729994e-02
            -6.02958838327382396e-01 -5.02139307676214952e-01 1.48370881953289995e-02
            -2.25347624992566026e-01 -7.52793309411134870e-01 1.47614709123309994e-02
            -7.63888747257869705e-01 -2.45084093906073108e-01 1.31654615542190004e-02
            -5.01406983635098058e-01 -6.30278174345496711e-01 1.42269552899920004e-02
            -8.14590318312071182e-01 -1.02414667496028344e-01 1.12948370918710007e-02
            -3.76577165676810766e-01 -7.44258193563618775e-01 1.32624073080989998e-02
            -8.97075099105038853e-02 -8.38740331484917556e-01 1.32318606149549999e-02
            -7.28692435382443326e-01 -4.56246819427198524e-01 1.19587045362549993e-02
            -6.35145681044101851e-01 -5.95337224218106531e-01 1.19101599437899993e-02
            -2.46134915729779913e-01 -8.38552835449991596e-01 1.15687313960700002e-02
            -8.10613049334679370e-01 -3.31195940831167213e-01 1.08246537112860006e-02
            -8.64329102915658232e-01 -1.82703866281845984e-01 9.92974474843799984e-03
            -5.18299056778332545e-01 -7.19194708013316308e-01 1.11675565859830003e-02
            -3.92557793518652609e-01 -8.21643113621251198e-01 9.65808471410700004e-03
            -1.47148525709283495e-01 -9.10349762177830701e-01 8.54249027516700010e-03
            -7.48630999239305650e-01 -5.41904601479935799e-01 8.83271956823299988e-03
            -9.27028017121971137e-01 -8.79632471159868534e-02 8.45223553558799995e-03
            -8.35618919688791428e-01 -4.15684930783212947e-01 8.20610183142100012e-03
            -6.41552974427336831e-01 -6.79161816231979132e-01 8.53654735225200037e-03
            -8.98182944079939816e-01 -2.64756390895498017e-01 7.82744669730900053e-03
            -2.87250883888211073e-01 -8.99029216084602290e-01 7.04244263164499965e-03
            -5.21594843533790375e-01 -7.91992590510683492e-01 7.48864467749500023e-03
            -6.67644208727447147e-02 -9.60719825750045109e-01 5.59393254402100037e-03
            -4.10164775379315227e-01 -8.76790329774203792e-01 5.31464774268599997e-03
            -7.45695427994018001e-01 -6.17220828768574203e-01 5.60859577042399984e-03
            -9.56212171764805996e-01 -1.73733787837638487e-01 5.49557138464999966e-03
            -8.39277183103726898e-01 -4.93333155706183413e-01 5.12874570092299985e-03
            -9.11513026544148564e-01 -3.44951464831909194e-01 5.04166234001500024e-03
            -2.09116655722167227e-01 -9.52674216498498483e-01 4.80417333523300034e-03
            -6.33554827683826427e-01 -7.45131826075070780e-01 4.92955357600600042e-03
            -5.19523983143803880e-01 -8.41268061702491687e-01 3.48602570293200017e-03
            -3.39444264710095422e-01 -9.29721150164553922e-01 3.00351288049500011e-03
            -9.19944100697844130e-02 -9.88328121673905269e-01 2.24411252693399984e-03
            -7.32575755985332089e-01 -6.71136450425255160e-01 2.39669624140899980e-03
            -9.90823718512418350e-01 -8.54860139910962463e-02 2.40780314138600016e-03
            -9.61810650834861836e-01 -2.53784640031552211e-01 2.30224558693400017e-03
            -8.25422851388604495e-01 -5.55542598048378133e-01 2.02948103005200003e-03
            -9.04898777443705060e-01 -4.14932055916150733e-01 1.98388331462500019e-03
            -2.23305163996555311e-01 -9.72537036373102626e-01 1.15696656985899999e-03
            -6.26164145956925711e-01 -7.78678481791463706e-01 9.55709298755000021e-04
            -4.36004265640657307e-01 -8.99317902617156939e-01 7.72276072839000040e-04
            ];


    case 41

        subcase=1;

        switch subcase
            case 1

                xyw=[
                    0.00000000000000000e+00 -0.00000000000000000e+00 2.41247016136439986e-02
                    1.43775391348433340e-01 8.32515737896245972e-02 2.36845319169330000e-02
                    2.85124385207706998e-01 0.00000000000000000e+00 2.28166240372399999e-02
                    2.83934837615607949e-01 1.63620881834912163e-01 2.23912167912590013e-02
                    4.19502465187274631e-01 -8.13671771115957454e-02 2.11223901939000015e-02
                    4.20156318735434020e-01 8.01113039822219514e-02 2.11343825848080008e-02
                    4.16525601247713151e-01 2.38875147456483389e-01 2.03246831275370014e-02
                    5.46867952776554977e-01 3.89839320119160786e-04 1.90772215739429991e-02
                    5.42978580801988131e-01 -1.56577371437142787e-01 1.86517273083099989e-02
                    5.45442460817058605e-01 1.56655473919450755e-01 1.87795765898300000e-02
                    5.35795667179719493e-01 3.09628743785371474e-01 1.75880293201870000e-02
                    6.60036458181430086e-01 -7.07227902137946840e-02 1.62094028962570004e-02
                    6.62757603911267812e-01 8.00283704021159531e-02 1.65074721455550008e-02
                    6.53467710632923193e-01 -2.19162888367116365e-01 1.55095371346080004e-02
                    6.55066128687800564e-01 2.30600332984445433e-01 1.58890200556659998e-02
                    6.34664267245810043e-01 3.78474325464307815e-01 1.44004826739510001e-02
                    7.64405989122058371e-01 9.22424520065001294e-03 1.33042312864000006e-02
                    7.57256542249431175e-01 -1.28444628499247815e-01 1.24146200020390004e-02
                    7.61448435424302295e-01 1.52882645195981576e-01 1.35251710588320008e-02
                    7.50932860435740812e-01 -2.61605451614362217e-01 1.21673590644639992e-02
                    7.42510753446148031e-01 2.98415447746632145e-01 1.25437118681180002e-02
                    7.09805607146746342e-01 4.40765831339129055e-01 1.12479090679009994e-02
                    8.50460227057837748e-01 2.20474208047850839e-02 1.09660010448080006e-02
                    8.42390234415362649e-01 -1.21817845760308963e-01 1.10059983536070008e-02
                    8.45510607569734840e-01 1.58973643893993322e-01 9.76142963517699983e-03
                    8.24084414796401821e-01 2.78205826193723804e-01 7.62716690907899991e-03
                    8.32641040394276999e-01 -2.64200584856002452e-01 1.00408674037379998e-02
                    8.00281681538628775e-01 3.86585656947298462e-01 7.68628509164199971e-03
                    7.45606376688899064e-01 5.14821958340902497e-01 7.91685363748399935e-03
                    9.17587547389138503e-01 -1.67702908529191279e-02 8.07725815170000014e-03
                    9.06157057157261581e-01 -1.50456300886004923e-01 7.37305363953100028e-03
                    9.14533975948910505e-01 1.23299065332345997e-01 8.00777693347299972e-03
                    8.89517531170053943e-01 2.67362911390587432e-01 7.70211576070699983e-03
                    8.98114916472220415e-01 -2.73748489241632198e-01 6.79764573376000041e-03
                    8.46295592435859612e-01 4.12688188214134721e-01 6.64377831730300020e-03
                    7.84582358979460159e-01 5.39611344085968225e-01 4.80161503968000013e-03
                    9.58056212175569133e-01 -1.19343048332157531e-01 5.50042774811800028e-03
                    9.66616033373711936e-01 2.56130426487205173e-02 5.51329959327400039e-03
                    9.53600366973766711e-01 1.73651413920376335e-01 5.21056877770299971e-03
                    9.18298309743854513e-01 3.18173465520550591e-01 4.69987519767499969e-03
                    9.43117713421938841e-01 -2.54605363065779244e-01 4.06266832568900026e-03
                    7.82361143106071633e-01 5.85797473165958427e-01 3.08913655496700001e-03
                    8.65583276560931036e-01 4.58180128214283000e-01 3.91727066004500003e-03
                    9.87406209872988549e-01 -1.09032509172519554e-01 2.52617402652899987e-03
                    9.92956808246925338e-01 3.95476098577115520e-02 2.37466390718900008e-03
                    9.76689654804058338e-01 1.85860504084157219e-01 2.18193538418500008e-03
                    9.39098195644969547e-01 3.27600776833471607e-01 2.02795743930499987e-03
                    8.02316832785673006e-01 5.88626164295792731e-01 1.85341720648599995e-03
                    9.62456537224400943e-01 -2.56918702089745155e-01 1.57673687637899999e-03
                    8.82576340566872064e-01 4.62973479107656016e-01 1.34671394905300007e-03
                    -2.10282132632958006e-04 1.66138928241604977e-01 2.36845319169330000e-02
                    1.42562192603853527e-01 2.46924960828294265e-01 2.28166240372399999e-02
                    2.67578549158316455e-04 3.27705223311981986e-01 2.23912167912590013e-02
                    2.80217275006507005e-01 3.22616203246579025e-01 2.11223901939000015e-02
                    1.40699734988815356e-01 4.03921697576548500e-01 2.11343825848080008e-02
                    1.39085459378832902e-03 4.80159325735348530e-01 2.03246831275370014e-02
                    2.73096365633660298e-01 4.73796459280144966e-01 1.90772215739429991e-02
                    4.07089271723351742e-01 3.91944558966771772e-01 1.86517273083099989e-02
                    1.37053610352394439e-01 5.50694764329996311e-01 1.87795765898300000e-02
                    -2.48524270135073349e-04 6.18827030907954989e-01 1.75880293201870000e-02
                    3.91265962042378790e-01 5.36246945102126338e-01 1.62094028962570004e-02
                    2.62072200163930957e-01 6.13979106739520719e-01 1.65074721455550008e-02
                    5.16534484209157485e-01 4.56338193777411805e-01 1.55095371346080004e-02
                    1.27827317858219991e-01 6.82604075094584317e-01 1.58890200556659998e-02
                    -1.04362469093650118e-02 7.38872541041261433e-01 1.44004826739510001e-02
                    3.74214563886529628e-01 6.66607127984998771e-01 1.33042312864000006e-02
                    4.89864582384718972e-01 5.91581088520347409e-01 1.24146200020390004e-02
                    2.48323963174668177e-01 7.35875011347351338e-01 1.35251710588320008e-02
                    6.02023397084408884e-01 5.19524207866684740e-01 1.21673590644639992e-02
                    1.12820018092782887e-01 7.92240898940804184e-01 1.25437118681180002e-02
                    -2.68116034864798493e-02 8.35092603207284290e-01 1.12479090679009994e-02
                    4.06136487024049475e-01 7.47543871942761839e-01 1.09660010448080006e-02
                    5.26692466270403559e-01 6.68622420023477826e-01 1.10059983536070008e-02
                    2.85080089640488421e-01 8.11720487271602398e-01 9.76142963517699983e-03
                    1.71108894433597974e-01 8.52780951173378488e-01 7.62716690907899991e-03
                    6.45124938377143020e-01 5.88988000786947574e-01 1.00408674037379998e-02
                    6.53478411142577598e-02 8.86357094869429818e-01 7.68628509164199971e-03
                    -7.30457060048259055e-02 9.03125042606707451e-01 7.91685363748399935e-03
                    4.73317271602051060e-01 7.86268980808791818e-01 8.07725815170000014e-03
                    5.83377507305346321e-01 7.09526880873733612e-01 7.37305363953100028e-03
                    3.50486865133766479e-01 8.53659188461916352e-01 8.00777693347299972e-03
                    2.13215692291010589e-01 9.04026234800176631e-01 7.70211576070699983e-03
                    6.86130604166974911e-01 6.40916088561865993e-01 6.79764573376000041e-03
                    6.57493413827155149e-02 9.39257576267323313e-01 6.64377831730300020e-03
                    -7.50259526589841519e-02 9.49273926280318481e-01 4.80161503968000013e-03
                    5.82382217708507066e-01 7.70029493831458245e-01 5.50042774811800028e-03
                    4.61126471084849887e-01 8.49920561931341467e-01 5.51329959327400039e-03
                    3.26413647628750858e-01 9.12667849817633270e-01 5.21056877770299971e-03
                    1.83602850920998484e-01 9.54356397250764377e-01 4.69987519767499969e-03
                    6.92053569065694596e-01 6.89461217049601394e-01 4.06266832568900026e-03
                    -1.16134921681416939e-01 9.70443361446669872e-01 3.08913655496700001e-03
                    3.59960077376853052e-02 9.78707170699879336e-01 3.91727066004500003e-03
                    5.88128027718256119e-01 8.00602607018257140e-01 2.52617402652899987e-03
                    4.62229169327728617e-01 8.79699625731406698e-01 2.37466390718900008e-03
                    3.27384909304967775e-01 9.38768304715847268e-01 2.18193538418500008e-03
                    1.85838502785181853e-01 9.77083282493408367e-01 2.02795743930499987e-03
                    -1.08606795219512692e-01 9.89139841224160876e-01 1.85341720648599995e-03
                    7.03726391329245970e-01 7.05052460229861788e-01 1.57673687637899999e-03
                    4.03413760977420696e-02 9.95820271263845647e-01 1.34671394905300007e-03
                    -1.43985673481066279e-01 8.28873544519804217e-02 2.36845319169330000e-02
                    -1.42562192603853444e-01 2.46924960828294293e-01 2.28166240372399999e-02
                    -2.83667259066449740e-01 1.64084341477069851e-01 2.23912167912590013e-02
                    -1.39285190180767599e-01 4.03983380358174826e-01 2.11223901939000015e-02
                    -2.79456583746618581e-01 3.23810393594326618e-01 2.11343825848080008e-02
                    -4.15134746653924913e-01 2.41284178278865197e-01 2.03246831275370014e-02
                    -2.73771587142894679e-01 4.73406619960025865e-01 1.90772215739429991e-02
                    -1.35889309078636306e-01 5.48521930403914615e-01 1.86517273083099989e-02
                    -4.08388850464664166e-01 3.94039290410545751e-01 1.87795765898300000e-02
                    -5.36044191449854379e-01 3.09198287122583682e-01 1.75880293201870000e-02
                    -2.68770496139051296e-01 6.06969735315921133e-01 1.62094028962570004e-02
                    -4.00685403747336799e-01 5.33950736337404863e-01 1.65074721455550008e-02
                    -1.36933226423765653e-01 6.75501082144528198e-01 1.55095371346080004e-02
                    -5.27238810829580462e-01 4.52003742110139106e-01 1.58890200556659998e-02
                    -6.45100514155175220e-01 3.60398215576953729e-01 1.44004826739510001e-02
                    -3.90191425235528633e-01 6.57382882784348865e-01 1.33042312864000006e-02
                    -2.67391959864712037e-01 7.20025717019595390e-01 1.24146200020390004e-02
                    -5.13124472249634200e-01 5.82992366151369734e-01 1.35251710588320008e-02
                    -1.48909463351331622e-01 7.81129659481046956e-01 1.21673590644639992e-02
                    -6.29690735353364950e-01 4.93825451194172316e-01 1.25437118681180002e-02
                    -7.36617210633226160e-01 3.94326771868155401e-01 1.12479090679009994e-02
                    -4.44323740033788162e-01 7.25496451137976894e-01 1.09660010448080006e-02
                    -3.15697768144959146e-01 7.90440265783787011e-01 1.10059983536070008e-02
                    -5.60430517929246585e-01 6.52746843377609021e-01 9.76142963517699983e-03
                    -6.52975520362803818e-01 5.74575124979654794e-01 7.62716690907899991e-03
                    -1.87516102017134006e-01 8.53188585642950081e-01 1.00408674037379998e-02
                    -7.34933840424370932e-01 4.99771437922131634e-01 7.68628509164199971e-03
                    -8.18652082693724914e-01 3.88303084265805232e-01 7.91685363748399935e-03
                    -4.44270275787087165e-01 8.03039271661711185e-01 8.07725815170000014e-03
                    -3.22779549851915259e-01 8.59983181759738646e-01 7.37305363953100028e-03
                    -5.64047110815143915e-01 7.30360123129570438e-01 8.00777693347299972e-03
                    -6.76301838879043493e-01 6.36663323409589310e-01 7.70211576070699983e-03
                    -2.11984312305245420e-01 9.14664577803498302e-01 6.79764573376000041e-03
                    -7.80546251053144124e-01 5.26569388053188869e-01 6.64377831730300020e-03
                    -8.59608311638444200e-01 4.09662582194350589e-01 4.80161503968000013e-03
                    -3.75673994467061845e-01 8.89372542163615831e-01 5.50042774811800028e-03
                    -5.05489562288862104e-01 8.24307519282621026e-01 5.51329959327400039e-03
                    -6.27186719345015686e-01 7.39016435897257185e-01 5.21056877770299971e-03
                    -7.34695458822856251e-01 6.36182931730214007e-01 4.69987519767499969e-03
                    -2.51064144356244190e-01 9.44066580115380805e-01 4.06266832568900026e-03
                    -8.98496064787488669e-01 3.84645888280711556e-01 3.08913655496700001e-03
                    -8.29587268823245738e-01 5.20527042485596336e-01 3.91727066004500003e-03
                    -3.99278182154732153e-01 9.09635116190776860e-01 2.52617402652899987e-03
                    -5.30727638919196387e-01 8.40152015873695368e-01 2.37466390718900008e-03
                    -6.49304745499090674e-01 7.52907800631690050e-01 2.18193538418500008e-03
                    -7.53259692859787777e-01 6.49482505659936926e-01 2.02795743930499987e-03
                    -9.10923628005185559e-01 4.00513676928368367e-01 1.85341720648599995e-03
                    -2.58730145895154806e-01 9.61971162319606998e-01 1.57673687637899999e-03
                    -8.42234964469130043e-01 5.32846792156189908e-01 1.34671394905300007e-03
                    -1.43775391348433368e-01 -8.32515737896245694e-02 2.36845319169330000e-02
                    -2.85124385207706998e-01 3.49176665703475298e-17 2.28166240372399999e-02
                    -2.83934837615607949e-01 -1.63620881834912163e-01 2.23912167912590013e-02
                    -4.19502465187274631e-01 8.13671771115958009e-02 2.11223901939000015e-02
                    -4.20156318735434020e-01 -8.01113039822219097e-02 2.11343825848080008e-02
                    -4.16525601247713151e-01 -2.38875147456483389e-01 2.03246831275370014e-02
                    -5.46867952776554977e-01 -3.89839320119098770e-04 1.90772215739429991e-02
                    -5.42978580801988131e-01 1.56577371437142732e-01 1.86517273083099989e-02
                    -5.45442460817058605e-01 -1.56655473919450672e-01 1.87795765898300000e-02
                    -5.35795667179719604e-01 -3.09628743785371363e-01 1.75880293201870000e-02
                    -6.60036458181430086e-01 7.07227902137947118e-02 1.62094028962570004e-02
                    -6.62757603911267812e-01 -8.00283704021158421e-02 1.65074721455550008e-02
                    -6.53467710632923082e-01 2.19162888367116587e-01 1.55095371346080004e-02
                    -6.55066128687800675e-01 -2.30600332984445239e-01 1.58890200556659998e-02
                    -6.34664267245810043e-01 -3.78474325464307815e-01 1.44004826739510001e-02
                    -7.64405989122058371e-01 -9.22424520064997651e-03 1.33042312864000006e-02
                    -7.57256542249431064e-01 1.28444628499248009e-01 1.24146200020390004e-02
                    -7.61448435424302295e-01 -1.52882645195981548e-01 1.35251710588320008e-02
                    -7.50932860435740701e-01 2.61605451614362439e-01 1.21673590644639992e-02
                    -7.42510753446148142e-01 -2.98415447746631923e-01 1.25437118681180002e-02
                    -7.09805607146746453e-01 -4.40765831339128944e-01 1.12479090679009994e-02
                    -8.50460227057837748e-01 -2.20474208047850180e-02 1.09660010448080006e-02
                    -8.42390234415362649e-01 1.21817845760309074e-01 1.10059983536070008e-02
                    -8.45510607569734840e-01 -1.58973643893993377e-01 9.76142963517699983e-03
                    -8.24084414796401932e-01 -2.78205826193723527e-01 7.62716690907899991e-03
                    -8.32641040394276888e-01 2.64200584856002674e-01 1.00408674037379998e-02
                    -8.00281681538628886e-01 -3.86585656947298351e-01 7.68628509164199971e-03
                    -7.45606376688899064e-01 -5.14821958340902608e-01 7.91685363748399935e-03
                    -9.17587547389138503e-01 1.67702908529193603e-02 8.07725815170000014e-03
                    -9.06157057157261470e-01 1.50456300886004951e-01 7.37305363953100028e-03
                    -9.14533975948910505e-01 -1.23299065332345914e-01 8.00777693347299972e-03
                    -8.89517531170053943e-01 -2.67362911390587432e-01 7.70211576070699983e-03
                    -8.98114916472220304e-01 2.73748489241632531e-01 6.79764573376000041e-03
                    -8.46295592435859612e-01 -4.12688188214134555e-01 6.64377831730300020e-03
                    -7.84582358979460159e-01 -5.39611344085968336e-01 4.80161503968000013e-03
                    -9.58056212175569022e-01 1.19343048332157656e-01 5.50042774811800028e-03
                    -9.66616033373711936e-01 -2.56130426487205937e-02 5.51329959327400039e-03
                    -9.53600366973766822e-01 -1.73651413920376141e-01 5.21056877770299971e-03
                    -9.18298309743854624e-01 -3.18173465520550591e-01 4.69987519767499969e-03
                    -9.43117713421938841e-01 2.54605363065779300e-01 4.06266832568900026e-03
                    -7.82361143106071633e-01 -5.85797473165958427e-01 3.08913655496700001e-03
                    -8.65583276560931036e-01 -4.58180128214282945e-01 3.91727066004500003e-03
                    -9.87406209872988549e-01 1.09032509172519554e-01 2.52617402652899987e-03
                    -9.92956808246925338e-01 -3.95476098577112675e-02 2.37466390718900008e-03
                    -9.76689654804058338e-01 -1.85860504084157246e-01 2.18193538418500008e-03
                    -9.39098195644969547e-01 -3.27600776833471552e-01 2.02795743930499987e-03
                    -8.02316832785673228e-01 -5.88626164295792620e-01 1.85341720648599995e-03
                    -9.62456537224400943e-01 2.56918702089745321e-01 1.57673687637899999e-03
                    -8.82576340566872175e-01 -4.62973479107655850e-01 1.34671394905300007e-03
                    2.10282132632863870e-04 -1.66138928241604977e-01 2.36845319169330000e-02
                    -1.42562192603853638e-01 -2.46924960828294238e-01 2.28166240372399999e-02
                    -2.67578549158502071e-04 -3.27705223311981986e-01 2.23912167912590013e-02
                    -2.80217275006507116e-01 -3.22616203246578914e-01 2.11223901939000015e-02
                    -1.40699734988815689e-01 -4.03921697576548389e-01 2.11343825848080008e-02
                    -1.39085459378860093e-03 -4.80159325735348530e-01 2.03246831275370014e-02
                    -2.73096365633660521e-01 -4.73796459280144855e-01 1.90772215739429991e-02
                    -4.07089271723351798e-01 -3.91944558966771717e-01 1.86517273083099989e-02
                    -1.37053610352394384e-01 -5.50694764329996311e-01 1.87795765898300000e-02
                    2.48524270134997563e-04 -6.18827030907954989e-01 1.75880293201870000e-02
                    -3.91265962042378956e-01 -5.36246945102126227e-01 1.62094028962570004e-02
                    -2.62072200163931124e-01 -6.13979106739520608e-01 1.65074721455550008e-02
                    -5.16534484209157707e-01 -4.56338193777411527e-01 1.55095371346080004e-02
                    -1.27827317858220380e-01 -6.82604075094584206e-01 1.58890200556659998e-02
                    1.04362469093649199e-02 -7.38872541041261433e-01 1.44004826739510001e-02
                    -3.74214563886529850e-01 -6.66607127984998660e-01 1.33042312864000006e-02
                    -4.89864582384718972e-01 -5.91581088520347409e-01 1.24146200020390004e-02
                    -2.48323963174668455e-01 -7.35875011347351227e-01 1.35251710588320008e-02
                    -6.02023397084409106e-01 -5.19524207866684407e-01 1.21673590644639992e-02
                    -1.12820018092782998e-01 -7.92240898940804184e-01 1.25437118681180002e-02
                    2.68116034864793740e-02 -8.35092603207284290e-01 1.12479090679009994e-02
                    -4.06136487024050086e-01 -7.47543871942761506e-01 1.09660010448080006e-02
                    -5.26692466270403448e-01 -6.68622420023477937e-01 1.10059983536070008e-02
                    -2.85080089640488921e-01 -8.11720487271602176e-01 9.76142963517699983e-03
                    -1.71108894433598446e-01 -8.52780951173378376e-01 7.62716690907899991e-03
                    -6.45124938377143242e-01 -5.88988000786947130e-01 1.00408674037379998e-02
                    -6.53478411142582594e-02 -8.86357094869429818e-01 7.68628509164199971e-03
                    7.30457060048257945e-02 -9.03125042606707451e-01 7.91685363748399935e-03
                    -4.73317271602051115e-01 -7.86268980808791818e-01 8.07725815170000014e-03
                    -5.83377507305346543e-01 -7.09526880873733390e-01 7.37305363953100028e-03
                    -3.50486865133767145e-01 -8.53659188461916019e-01 8.00777693347299972e-03
                    -2.13215692291011116e-01 -9.04026234800176520e-01 7.70211576070699983e-03
                    -6.86130604166975244e-01 -6.40916088561865549e-01 6.79764573376000041e-03
                    -6.57493413827154177e-02 -9.39257576267323313e-01 6.64377831730300020e-03
                    7.50259526589840409e-02 -9.49273926280318592e-01 4.80161503968000013e-03
                    -5.82382217708507732e-01 -7.70029493831457690e-01 5.50042774811800028e-03
                    -4.61126471084849998e-01 -8.49920561931341467e-01 5.51329959327400039e-03
                    -3.26413647628751191e-01 -9.12667849817633159e-01 5.21056877770299971e-03
                    -1.83602850920999039e-01 -9.54356397250764266e-01 4.69987519767499969e-03
                    -6.92053569065694818e-01 -6.89461217049601172e-01 4.06266832568900026e-03
                    1.16134921681416384e-01 -9.70443361446669872e-01 3.08913655496700001e-03
                    -3.59960077376860754e-02 -9.78707170699879225e-01 3.91727066004500003e-03
                    -5.88128027718256674e-01 -8.00602607018256807e-01 2.52617402652899987e-03
                    -4.62229169327729117e-01 -8.79699625731406476e-01 2.37466390718900008e-03
                    -3.27384909304967886e-01 -9.38768304715847157e-01 2.18193538418500008e-03
                    -1.85838502785182630e-01 -9.77083282493408145e-01 2.02795743930499987e-03
                    1.08606795219512567e-01 -9.89139841224160876e-01 1.85341720648599995e-03
                    -7.03726391329246304e-01 -7.05052460229861566e-01 1.57673687637899999e-03
                    -4.03413760977424166e-02 -9.95820271263845647e-01 1.34671394905300007e-03
                    1.43985673481066223e-01 -8.28873544519805050e-02 2.36845319169330000e-02
                    1.42562192603853305e-01 -2.46924960828294404e-01 2.28166240372399999e-02
                    2.83667259066449517e-01 -1.64084341477070156e-01 2.23912167912590013e-02
                    1.39285190180767349e-01 -4.03983380358174937e-01 2.11223901939000015e-02
                    2.79456583746618248e-01 -3.23810393594326895e-01 2.11343825848080008e-02
                    4.15134746653924636e-01 -2.41284178278865641e-01 2.03246831275370014e-02
                    2.73771587142894401e-01 -4.73406619960026087e-01 1.90772215739429991e-02
                    1.35889309078636111e-01 -5.48521930403914615e-01 1.86517273083099989e-02
                    4.08388850464664055e-01 -3.94039290410545806e-01 1.87795765898300000e-02
                    5.36044191449854379e-01 -3.09198287122583793e-01 1.75880293201870000e-02
                    2.68770496139050907e-01 -6.06969735315921355e-01 1.62094028962570004e-02
                    4.00685403747336522e-01 -5.33950736337405196e-01 1.65074721455550008e-02
                    1.36933226423765098e-01 -6.75501082144528420e-01 1.55095371346080004e-02
                    5.27238810829580240e-01 -4.52003742110139384e-01 1.58890200556659998e-02
                    6.45100514155174998e-01 -3.60398215576954117e-01 1.44004826739510001e-02
                    3.90191425235528300e-01 -6.57382882784349087e-01 1.33042312864000006e-02
                    2.67391959864711926e-01 -7.20025717019595501e-01 1.24146200020390004e-02
                    5.13124472249633867e-01 -5.82992366151370178e-01 1.35251710588320008e-02
                    1.48909463351331178e-01 -7.81129659481047067e-01 1.21673590644639992e-02
                    6.29690735353364950e-01 -4.93825451194172427e-01 1.25437118681180002e-02
                    7.36617210633225827e-01 -3.94326771868155790e-01 1.12479090679009994e-02
                    4.44323740033787384e-01 -7.25496451137977227e-01 1.09660010448080006e-02
                    3.15697768144959034e-01 -7.90440265783787011e-01 1.10059983536070008e-02
                    5.60430517929245919e-01 -6.52746843377609687e-01 9.76142963517699983e-03
                    6.52975520362803263e-01 -5.74575124979655460e-01 7.62716690907899991e-03
                    1.87516102017133701e-01 -8.53188585642950081e-01 1.00408674037379998e-02
                    7.34933840424370599e-01 -4.99771437922132022e-01 7.68628509164199971e-03
                    8.18652082693724914e-01 -3.88303084265805343e-01 7.91685363748399935e-03
                    4.44270275787087110e-01 -8.03039271661711185e-01 8.07725815170000014e-03
                    3.22779549851914760e-01 -8.59983181759738757e-01 7.37305363953100028e-03
                    5.64047110815143249e-01 -7.30360123129570993e-01 8.00777693347299972e-03
                    6.76301838879042938e-01 -6.36663323409590087e-01 7.70211576070699983e-03
                    2.11984312305244699e-01 -9.14664577803498413e-01 6.79764573376000041e-03
                    7.80546251053144013e-01 -5.26569388053188980e-01 6.64377831730300020e-03
                    8.59608311638444200e-01 -4.09662582194350700e-01 4.80161503968000013e-03
                    3.75673994467061734e-01 -8.89372542163615942e-01 5.50042774811800028e-03
                    5.05489562288861771e-01 -8.24307519282621359e-01 5.51329959327400039e-03
                    6.27186719345015353e-01 -7.39016435897257518e-01 5.21056877770299971e-03
                    7.34695458822855696e-01 -6.36182931730214674e-01 4.69987519767499969e-03
                    2.51064144356243635e-01 -9.44066580115380916e-01 4.06266832568900026e-03
                    8.98496064787488335e-01 -3.84645888280712500e-01 3.08913655496700001e-03
                    8.29587268823245183e-01 -5.20527042485597224e-01 3.91727066004500003e-03
                    3.99278182154731431e-01 -9.09635116190777193e-01 2.52617402652899987e-03
                    5.30727638919195943e-01 -8.40152015873695701e-01 2.37466390718900008e-03
                    6.49304745499090230e-01 -7.52907800631690383e-01 2.18193538418500008e-03
                    7.53259692859787111e-01 -6.49482505659937703e-01 2.02795743930499987e-03
                    9.10923628005185559e-01 -4.00513676928368478e-01 1.85341720648599995e-03
                    2.58730145895153807e-01 -9.61971162319607331e-01 1.57673687637899999e-03
                    8.42234964469129710e-01 -5.32846792156190352e-01 1.34671394905300007e-03
                    ];

            case 2

                xyw=[
                    -0.00000000000000000e+00 -0.00000000000000000e+00 2.40778603874480003e-02
                    8.33282364376118495e-02 -1.43116656624509120e-01 2.36243830632670015e-02
                    1.66726744255912995e-01 0.00000000000000000e+00 2.36685814982959992e-02
                    -2.93037462118654143e-14 -2.83370138098251012e-01 2.26758208578560008e-02
                    2.47302635693990164e-01 -1.42856672171941951e-01 2.28224301993920006e-02
                    1.63678173609610789e-01 -2.83118320002205770e-01 2.22898794144500006e-02
                    3.28086221275448009e-01 -0.00000000000000000e+00 2.24576375550980013e-02
                    8.01182052644353881e-02 -4.17707400429675801e-01 2.08653844030809998e-02
                    4.03359460959417215e-01 -1.42268123621058035e-01 2.12523540561790000e-02
                    3.22116346595061609e-01 -2.82304306525009874e-01 2.11510672895250003e-02
                    4.78949373564846015e-01 0.00000000000000000e+00 2.04892348945199983e-02
                    2.37772739766760333e-01 -4.16947309082725037e-01 2.01510098509520008e-02
                    -5.62238356016992093e-14 -5.43690077837753027e-01 1.86257198569059999e-02
                    4.70496159565388594e-01 -2.80453724921636727e-01 1.93129815897279990e-02
                    1.53403371407475342e-01 -5.43205280380892663e-01 1.82956497806890002e-02
                    5.46757609609815010e-01 -1.40886866029216129e-01 1.89693847127209991e-02
                    3.87975553692327724e-01 -4.15035325666648192e-01 1.87664983122570010e-02
                    6.14848201706059028e-01 -0.00000000000000000e+00 1.78122584582680009e-02
                    3.01946430958207945e-01 -5.41535376507241262e-01 1.73530084762729998e-02
                    7.25007094462424145e-02 -6.58336772263512060e-01 1.56713003370569995e-02
                    6.05168114262100487e-01 -2.75390512143867672e-01 1.68355107793669995e-02
                    5.27048853408047413e-01 -4.10012340901986627e-01 1.67804420684610003e-02
                    6.73941624042249554e-01 -1.36851871151165622e-01 1.60153180298500016e-02
                    2.14946006360511088e-01 -6.56978339032438852e-01 1.50254535722339991e-02
                    4.41943540646229871e-01 -5.37101942221313799e-01 1.59153038500120005e-02
                    7.32653198728905997e-01 -0.00000000000000000e+00 1.43897035204960007e-02
                    3.50316062392283245e-01 -6.54048525814154069e-01 1.41195823704680003e-02
                    -7.86114482625405425e-14 -7.60180517166770042e-01 1.29669523584940004e-02
                    6.54975390408357505e-01 -3.92905594208993625e-01 1.40518024875039994e-02
                    7.25897356033784913e-01 -2.57749590384032667e-01 1.37768085876290008e-02
                    1.35806977726169564e-01 -7.58422749136705376e-01 1.22713119645910008e-02
                    5.75476950238713925e-01 -5.20573394717303572e-01 1.39029750713000003e-02
                    7.85004348097258942e-01 -1.20358495863269382e-01 1.25402443028579998e-02
                    4.83689443421676135e-01 -6.38521723625323112e-01 1.25928847183679991e-02
                    2.61785345940408254e-01 -7.58017644476261965e-01 1.15408456998300005e-02
                    8.37629675645187977e-01 -0.00000000000000000e+00 1.08566584497579999e-02
                    3.88235759594040386e-01 -7.43452748895638948e-01 1.03642013645580001e-02
                    7.54725673305997824e-01 -3.71649887551330382e-01 7.90211490587799936e-03
                    7.27875097777751223e-02 -8.45032246180391966e-01 1.10906218547939996e-02
                    6.92810262541782795e-01 -4.98311728019836087e-01 1.14283415518089994e-02
                    8.23622827854065487e-01 -2.33367041381119023e-01 1.00844944515780001e-02
                    6.04149320105910603e-01 -6.18243096627734579e-01 1.06483870533870002e-02
                    2.09024116835555968e-01 -8.43942873403710325e-01 9.91883252894800067e-03
                    7.94383631002849699e-01 -3.61762469300777156e-01 5.25794719661800032e-03
                    5.05050305495659435e-01 -7.25236590572820083e-01 9.00674779239699942e-03
                    8.77324705443737085e-01 -1.13390529948523444e-01 9.09129595141400064e-03
                    3.39712450321614068e-01 -8.26846628440769393e-01 8.35594255284700070e-03
                    -9.44418656646130263e-14 -9.13262226684232958e-01 8.19445670658200051e-03
                    9.18611589478634971e-01 -0.00000000000000000e+00 7.52913538425200018e-03
                    7.81196534992100067e-01 -4.83617700515943105e-01 8.12400808234300074e-03
                    8.84590806573783195e-01 -2.55353346018845684e-01 6.89230560377900037e-03
                    6.99845394055578551e-01 -6.01805326401524732e-01 8.03304209343299984e-03
                    1.35706988672543494e-01 -9.13552902317762761e-01 7.81168365210500003e-03
                    8.48900536293248797e-01 -3.69126476872350839e-01 6.35124129113999979e-03
                    4.51815013492781248e-01 -8.08781114349203523e-01 6.67973841886999962e-03
                    6.02472725075903992e-01 -7.09262876621667870e-01 7.10695862987999975e-03
                    2.74294068917462441e-01 -8.97587316234321086e-01 6.88823123455899970e-03
                    9.33406738298651728e-01 -1.43991741473502355e-01 6.60173622755500043e-03
                    3.98281557701155364e-01 -8.72648125801423702e-01 4.66419393727899971e-03
                    5.45070241146952794e-01 -7.90904344898425760e-01 4.90375120275199968e-03
                    9.64003724489905012e-01 -0.00000000000000000e+00 5.34984771993800030e-03
                    5.09720379082743355e-02 -9.63728061332782771e-01 4.28261776785299978e-03
                    7.86782088858422934e-01 -5.62294173971507494e-01 5.62929146880000036e-03
                    8.64625979721695193e-01 -4.34713705476526879e-01 5.43648139867700022e-03
                    9.24244742598984703e-01 -2.91320059312684787e-01 5.53601583940400033e-03
                    6.89450251448256668e-01 -6.81680282103836910e-01 5.25861437362399979e-03
                    1.77376097724339321e-01 -9.55106950097069030e-01 4.64986914259600016e-03
                    3.11762766926646340e-01 -9.28863195230936189e-01 3.52724892876299986e-03
                    9.70548332676287262e-01 -1.42817000235277713e-01 3.93705547530200008e-03
                    4.71015240470604002e-01 -8.63832466567013402e-01 3.16988483898999985e-03
                    6.05620017736163696e-01 -7.76100660816042121e-01 2.76228288478600013e-03
                    9.91565361412792945e-01 -0.00000000000000000e+00 2.46556745458399989e-03
                    7.40289715445972690e-02 -9.90693239741754383e-01 2.49096311459699998e-03
                    7.55715380409447968e-01 -6.45147335749373774e-01 2.36384376675300009e-03
                    9.12089908253746806e-01 -3.94488725953920594e-01 2.34718358962999986e-03
                    8.43257211940908857e-01 -5.26002453225995370e-01 2.35659053874399992e-03
                    9.60608006066268838e-01 -2.57986117049701480e-01 1.99100374207599991e-03
                    2.23329052372056663e-01 -9.70194673097432903e-01 1.82066169141500002e-03
                    3.79684066987245217e-01 -9.21082693543939457e-01 1.58140143397599996e-03
                    6.59642882473617242e-01 -7.46722135420716970e-01 1.35099198174799991e-03
                    5.31373272478381953e-01 -8.44078282981892358e-01 1.18611055353199995e-03
                    9.91979719359388223e-01 -1.14645109803836770e-01 8.60165845832999969e-04
                    8.33282364376118495e-02 1.43116656624509120e-01 2.36243830632670015e-02
                    -2.93037462118654143e-14 2.83370138098251012e-01 2.26758208578560008e-02
                    2.47302635693990164e-01 1.42856672171941951e-01 2.28224301993920006e-02
                    1.63678173609610789e-01 2.83118320002205770e-01 2.22898794144500006e-02
                    8.01182052644353881e-02 4.17707400429675801e-01 2.08653844030809998e-02
                    4.03359460959417215e-01 1.42268123621058035e-01 2.12523540561790000e-02
                    3.22116346595061609e-01 2.82304306525009874e-01 2.11510672895250003e-02
                    2.37772739766760333e-01 4.16947309082725037e-01 2.01510098509520008e-02
                    -5.62238356016992093e-14 5.43690077837753027e-01 1.86257198569059999e-02
                    4.70496159565388594e-01 2.80453724921636727e-01 1.93129815897279990e-02
                    1.53403371407475342e-01 5.43205280380892663e-01 1.82956497806890002e-02
                    5.46757609609815010e-01 1.40886866029216129e-01 1.89693847127209991e-02
                    3.87975553692327724e-01 4.15035325666648192e-01 1.87664983122570010e-02
                    3.01946430958207945e-01 5.41535376507241262e-01 1.73530084762729998e-02
                    7.25007094462424145e-02 6.58336772263512060e-01 1.56713003370569995e-02
                    6.05168114262100487e-01 2.75390512143867672e-01 1.68355107793669995e-02
                    5.27048853408047413e-01 4.10012340901986627e-01 1.67804420684610003e-02
                    6.73941624042249554e-01 1.36851871151165622e-01 1.60153180298500016e-02
                    2.14946006360511088e-01 6.56978339032438852e-01 1.50254535722339991e-02
                    4.41943540646229871e-01 5.37101942221313799e-01 1.59153038500120005e-02
                    3.50316062392283245e-01 6.54048525814154069e-01 1.41195823704680003e-02
                    -7.86114482625405425e-14 7.60180517166770042e-01 1.29669523584940004e-02
                    6.54975390408357505e-01 3.92905594208993625e-01 1.40518024875039994e-02
                    7.25897356033784913e-01 2.57749590384032667e-01 1.37768085876290008e-02
                    1.35806977726169564e-01 7.58422749136705376e-01 1.22713119645910008e-02
                    5.75476950238713925e-01 5.20573394717303572e-01 1.39029750713000003e-02
                    7.85004348097258942e-01 1.20358495863269382e-01 1.25402443028579998e-02
                    4.83689443421676135e-01 6.38521723625323112e-01 1.25928847183679991e-02
                    2.61785345940408254e-01 7.58017644476261965e-01 1.15408456998300005e-02
                    3.88235759594040386e-01 7.43452748895638948e-01 1.03642013645580001e-02
                    7.54725673305997824e-01 3.71649887551330382e-01 7.90211490587799936e-03
                    7.27875097777751223e-02 8.45032246180391966e-01 1.10906218547939996e-02
                    6.92810262541782795e-01 4.98311728019836087e-01 1.14283415518089994e-02
                    8.23622827854065487e-01 2.33367041381119023e-01 1.00844944515780001e-02
                    6.04149320105910603e-01 6.18243096627734579e-01 1.06483870533870002e-02
                    2.09024116835555968e-01 8.43942873403710325e-01 9.91883252894800067e-03
                    7.94383631002849699e-01 3.61762469300777156e-01 5.25794719661800032e-03
                    5.05050305495659435e-01 7.25236590572820083e-01 9.00674779239699942e-03
                    8.77324705443737085e-01 1.13390529948523444e-01 9.09129595141400064e-03
                    3.39712450321614068e-01 8.26846628440769393e-01 8.35594255284700070e-03
                    -9.44418656646130263e-14 9.13262226684232958e-01 8.19445670658200051e-03
                    7.81196534992100067e-01 4.83617700515943105e-01 8.12400808234300074e-03
                    8.84590806573783195e-01 2.55353346018845684e-01 6.89230560377900037e-03
                    6.99845394055578551e-01 6.01805326401524732e-01 8.03304209343299984e-03
                    1.35706988672543494e-01 9.13552902317762761e-01 7.81168365210500003e-03
                    8.48900536293248797e-01 3.69126476872350839e-01 6.35124129113999979e-03
                    4.51815013492781248e-01 8.08781114349203523e-01 6.67973841886999962e-03
                    6.02472725075903992e-01 7.09262876621667870e-01 7.10695862987999975e-03
                    2.74294068917462441e-01 8.97587316234321086e-01 6.88823123455899970e-03
                    9.33406738298651728e-01 1.43991741473502355e-01 6.60173622755500043e-03
                    3.98281557701155364e-01 8.72648125801423702e-01 4.66419393727899971e-03
                    5.45070241146952794e-01 7.90904344898425760e-01 4.90375120275199968e-03
                    5.09720379082743355e-02 9.63728061332782771e-01 4.28261776785299978e-03
                    7.86782088858422934e-01 5.62294173971507494e-01 5.62929146880000036e-03
                    8.64625979721695193e-01 4.34713705476526879e-01 5.43648139867700022e-03
                    9.24244742598984703e-01 2.91320059312684787e-01 5.53601583940400033e-03
                    6.89450251448256668e-01 6.81680282103836910e-01 5.25861437362399979e-03
                    1.77376097724339321e-01 9.55106950097069030e-01 4.64986914259600016e-03
                    3.11762766926646340e-01 9.28863195230936189e-01 3.52724892876299986e-03
                    9.70548332676287262e-01 1.42817000235277713e-01 3.93705547530200008e-03
                    4.71015240470604002e-01 8.63832466567013402e-01 3.16988483898999985e-03
                    6.05620017736163696e-01 7.76100660816042121e-01 2.76228288478600013e-03
                    7.40289715445972690e-02 9.90693239741754383e-01 2.49096311459699998e-03
                    7.55715380409447968e-01 6.45147335749373774e-01 2.36384376675300009e-03
                    9.12089908253746806e-01 3.94488725953920594e-01 2.34718358962999986e-03
                    8.43257211940908857e-01 5.26002453225995370e-01 2.35659053874399992e-03
                    9.60608006066268838e-01 2.57986117049701480e-01 1.99100374207599991e-03
                    2.23329052372056663e-01 9.70194673097432903e-01 1.82066169141500002e-03
                    3.79684066987245217e-01 9.21082693543939457e-01 1.58140143397599996e-03
                    6.59642882473617242e-01 7.46722135420716970e-01 1.35099198174799991e-03
                    5.31373272478381953e-01 8.44078282981892358e-01 1.18611055353199995e-03
                    9.91979719359388223e-01 1.14645109803836770e-01 8.60165845832999969e-04
                    -8.33282364376117940e-02 1.43116656624509148e-01 2.36243830632670015e-02
                    -1.66726744255912995e-01 2.04181373685263217e-17 2.36685814982959992e-02
                    -2.47302635693990136e-01 1.42856672171941979e-01 2.28224301993920006e-02
                    -1.63678173609610761e-01 2.83118320002205826e-01 2.22898794144500006e-02
                    -3.28086221275448009e-01 4.01789740729327658e-17 2.24576375550980013e-02
                    -8.01182052644353465e-02 4.17707400429675801e-01 2.08653844030809998e-02
                    -4.03359460959417215e-01 1.42268123621058035e-01 2.12523540561790000e-02
                    -3.22116346595061498e-01 2.82304306525009929e-01 2.11510672895250003e-02
                    -4.78949373564846015e-01 5.86543817289818602e-17 2.04892348945199983e-02
                    -2.37772739766760194e-01 4.16947309082725148e-01 2.01510098509520008e-02
                    -4.70496159565388539e-01 2.80453724921636782e-01 1.93129815897279990e-02
                    -1.53403371407475259e-01 5.43205280380892663e-01 1.82956497806890002e-02
                    -5.46757609609815010e-01 1.40886866029216185e-01 1.89693847127209991e-02
                    -3.87975553692327613e-01 4.15035325666648303e-01 1.87664983122570010e-02
                    -6.14848201706059028e-01 7.52971882180831332e-17 1.78122584582680009e-02
                    -3.01946430958207834e-01 5.41535376507241262e-01 1.73530084762729998e-02
                    -7.25007094462423313e-02 6.58336772263512060e-01 1.56713003370569995e-02
                    -6.05168114262100487e-01 2.75390512143867672e-01 1.68355107793669995e-02
                    -5.27048853408047524e-01 4.10012340901986572e-01 1.67804420684610003e-02
                    -6.73941624042249443e-01 1.36851871151165844e-01 1.60153180298500016e-02
                    -2.14946006360511033e-01 6.56978339032438852e-01 1.50254535722339991e-02
                    -4.41943540646229704e-01 5.37101942221314022e-01 1.59153038500120005e-02
                    -7.32653198728905997e-01 8.97241394708424345e-17 1.43897035204960007e-02
                    -3.50316062392283134e-01 6.54048525814154069e-01 1.41195823704680003e-02
                    -6.54975390408357505e-01 3.92905594208993791e-01 1.40518024875039994e-02
                    -7.25897356033784913e-01 2.57749590384032945e-01 1.37768085876290008e-02
                    -1.35806977726169481e-01 7.58422749136705376e-01 1.22713119645910008e-02
                    -5.75476950238713703e-01 5.20573394717303795e-01 1.39029750713000003e-02
                    -7.85004348097258942e-01 1.20358495863269382e-01 1.25402443028579998e-02
                    -4.83689443421676135e-01 6.38521723625323001e-01 1.25928847183679991e-02
                    -2.61785345940408143e-01 7.58017644476262076e-01 1.15408456998300005e-02
                    -8.37629675645187977e-01 1.02580050114971508e-16 1.08566584497579999e-02
                    -3.88235759594040331e-01 7.43452748895639060e-01 1.03642013645580001e-02
                    -7.54725673305997824e-01 3.71649887551330327e-01 7.90211490587799936e-03
                    -7.27875097777750252e-02 8.45032246180391966e-01 1.10906218547939996e-02
                    -6.92810262541782684e-01 4.98311728019836309e-01 1.14283415518089994e-02
                    -8.23622827854065487e-01 2.33367041381119023e-01 1.00844944515780001e-02
                    -6.04149320105910381e-01 6.18243096627734801e-01 1.06483870533870002e-02
                    -2.09024116835555884e-01 8.43942873403710325e-01 9.91883252894800067e-03
                    -7.94383631002849699e-01 3.61762469300777267e-01 5.25794719661800032e-03
                    -5.05050305495659324e-01 7.25236590572820083e-01 9.00674779239699942e-03
                    -8.77324705443737085e-01 1.13390529948523541e-01 9.09129595141400064e-03
                    -3.39712450321614012e-01 8.26846628440769393e-01 8.35594255284700070e-03
                    -9.18611589478634971e-01 1.12497474271467272e-16 7.52913538425200018e-03
                    -7.81196534992099956e-01 4.83617700515943327e-01 8.12400808234300074e-03
                    -8.84590806573783306e-01 2.55353346018845628e-01 6.89230560377900037e-03
                    -6.99845394055578440e-01 6.01805326401524843e-01 8.03304209343299984e-03
                    -1.35706988672543355e-01 9.13552902317762761e-01 7.81168365210500003e-03
                    -8.48900536293248686e-01 3.69126476872351172e-01 6.35124129113999979e-03
                    -4.51815013492781137e-01 8.08781114349203634e-01 6.67973841886999962e-03
                    -6.02472725075903770e-01 7.09262876621668092e-01 7.10695862987999975e-03
                    -2.74294068917462330e-01 8.97587316234321086e-01 6.88823123455899970e-03
                    -9.33406738298651728e-01 1.43991741473502299e-01 6.60173622755500043e-03
                    -3.98281557701155253e-01 8.72648125801423813e-01 4.66419393727899971e-03
                    -5.45070241146952461e-01 7.90904344898425982e-01 4.90375120275199968e-03
                    -9.64003724489905012e-01 1.18056407556268902e-16 5.34984771993800030e-03
                    -5.09720379082742245e-02 9.63728061332782771e-01 4.28261776785299978e-03
                    -7.86782088858422823e-01 5.62294173971507605e-01 5.62929146880000036e-03
                    -8.64625979721695193e-01 4.34713705476526824e-01 5.43648139867700022e-03
                    -9.24244742598984814e-01 2.91320059312684676e-01 5.53601583940400033e-03
                    -6.89450251448256668e-01 6.81680282103837021e-01 5.25861437362399979e-03
                    -1.77376097724339210e-01 9.55106950097069030e-01 4.64986914259600016e-03
                    -3.11762766926646229e-01 9.28863195230936189e-01 3.52724892876299986e-03
                    -9.70548332676287151e-01 1.42817000235277852e-01 3.93705547530200008e-03
                    -4.71015240470604113e-01 8.63832466567013402e-01 3.16988483898999985e-03
                    -6.05620017736163585e-01 7.76100660816042232e-01 2.76228288478600013e-03
                    -9.91565361412792945e-01 1.21431734599956522e-16 2.46556745458399989e-03
                    -7.40289715445971441e-02 9.90693239741754383e-01 2.49096311459699998e-03
                    -7.55715380409447746e-01 6.45147335749373996e-01 2.36384376675300009e-03
                    -9.12089908253746806e-01 3.94488725953920705e-01 2.34718358962999986e-03
                    -8.43257211940908746e-01 5.26002453225995481e-01 2.35659053874399992e-03
                    -9.60608006066268838e-01 2.57986117049701424e-01 1.99100374207599991e-03
                    -2.23329052372056552e-01 9.70194673097433014e-01 1.82066169141500002e-03
                    -3.79684066987245106e-01 9.21082693543939457e-01 1.58140143397599996e-03
                    -6.59642882473617242e-01 7.46722135420716970e-01 1.35099198174799991e-03
                    -5.31373272478381620e-01 8.44078282981892580e-01 1.18611055353199995e-03
                    -9.91979719359388223e-01 1.14645109803836742e-01 8.60165845832999969e-04
                    -8.33282364376118218e-02 -1.43116656624509120e-01 2.36243830632670015e-02
                    -2.47302635693990164e-01 -1.42856672171941895e-01 2.28224301993920006e-02
                    -1.63678173609610844e-01 -2.83118320002205770e-01 2.22898794144500006e-02
                    -8.01182052644355408e-02 -4.17707400429675746e-01 2.08653844030809998e-02
                    -4.03359460959417271e-01 -1.42268123621057924e-01 2.12523540561790000e-02
                    -3.22116346595061609e-01 -2.82304306525009874e-01 2.11510672895250003e-02
                    -2.37772739766760305e-01 -4.16947309082725093e-01 2.01510098509520008e-02
                    -4.70496159565388650e-01 -2.80453724921636616e-01 1.93129815897279990e-02
                    -1.53403371407475536e-01 -5.43205280380892552e-01 1.82956497806890002e-02
                    -5.46757609609815010e-01 -1.40886866029216046e-01 1.89693847127209991e-02
                    -3.87975553692327724e-01 -4.15035325666648192e-01 1.87664983122570010e-02
                    -3.01946430958207945e-01 -5.41535376507241151e-01 1.73530084762729998e-02
                    -7.25007094462424978e-02 -6.58336772263512060e-01 1.56713003370569995e-02
                    -6.05168114262100598e-01 -2.75390512143867561e-01 1.68355107793669995e-02
                    -5.27048853408047524e-01 -4.10012340901986461e-01 1.67804420684610003e-02
                    -6.73941624042249554e-01 -1.36851871151165677e-01 1.60153180298500016e-02
                    -2.14946006360511199e-01 -6.56978339032438852e-01 1.50254535722339991e-02
                    -4.41943540646229871e-01 -5.37101942221313799e-01 1.59153038500120005e-02
                    -3.50316062392283301e-01 -6.54048525814154069e-01 1.41195823704680003e-02
                    -6.54975390408357505e-01 -3.92905594208993625e-01 1.40518024875039994e-02
                    -7.25897356033784913e-01 -2.57749590384032778e-01 1.37768085876290008e-02
                    -1.35806977726169509e-01 -7.58422749136705376e-01 1.22713119645910008e-02
                    -5.75476950238713925e-01 -5.20573394717303684e-01 1.39029750713000003e-02
                    -7.85004348097258942e-01 -1.20358495863269188e-01 1.25402443028579998e-02
                    -4.83689443421676302e-01 -6.38521723625322890e-01 1.25928847183679991e-02
                    -2.61785345940408309e-01 -7.58017644476261965e-01 1.15408456998300005e-02
                    -3.88235759594040775e-01 -7.43452748895638726e-01 1.03642013645580001e-02
                    -7.54725673305997935e-01 -3.71649887551330105e-01 7.90211490587799936e-03
                    -7.27875097777752195e-02 -8.45032246180391966e-01 1.10906218547939996e-02
                    -6.92810262541782795e-01 -4.98311728019836087e-01 1.14283415518089994e-02
                    -8.23622827854065598e-01 -2.33367041381118828e-01 1.00844944515780001e-02
                    -6.04149320105910603e-01 -6.18243096627734579e-01 1.06483870533870002e-02
                    -2.09024116835555718e-01 -8.43942873403710436e-01 9.91883252894800067e-03
                    -7.94383631002849699e-01 -3.61762469300777045e-01 5.25794719661800032e-03
                    -5.05050305495659768e-01 -7.25236590572819750e-01 9.00674779239699942e-03
                    -8.77324705443737085e-01 -1.13390529948523347e-01 9.09129595141400064e-03
                    -3.39712450321614012e-01 -8.26846628440769393e-01 8.35594255284700070e-03
                    -7.81196534992100067e-01 -4.83617700515943105e-01 8.12400808234300074e-03
                    -8.84590806573783306e-01 -2.55353346018845406e-01 6.89230560377900037e-03
                    -6.99845394055578662e-01 -6.01805326401524621e-01 8.03304209343299984e-03
                    -1.35706988672543383e-01 -9.13552902317762761e-01 7.81168365210500003e-03
                    -8.48900536293248686e-01 -3.69126476872350895e-01 6.35124129113999979e-03
                    -4.51815013492781692e-01 -8.08781114349203301e-01 6.67973841886999962e-03
                    -6.02472725075903881e-01 -7.09262876621667981e-01 7.10695862987999975e-03
                    -2.74294068917462774e-01 -8.97587316234320975e-01 6.88823123455899970e-03
                    -9.33406738298651839e-01 -1.43991741473502050e-01 6.60173622755500043e-03
                    -3.98281557701155253e-01 -8.72648125801423813e-01 4.66419393727899971e-03
                    -5.45070241146952683e-01 -7.90904344898425760e-01 4.90375120275199968e-03
                    -5.09720379082746686e-02 -9.63728061332782771e-01 4.28261776785299978e-03
                    -7.86782088858423045e-01 -5.62294173971507383e-01 5.62929146880000036e-03
                    -8.64625979721695304e-01 -4.34713705476526602e-01 5.43648139867700022e-03
                    -9.24244742598984814e-01 -2.91320059312684454e-01 5.53601583940400033e-03
                    -6.89450251448256779e-01 -6.81680282103836799e-01 5.25861437362399979e-03
                    -1.77376097724339238e-01 -9.55106950097069030e-01 4.64986914259600016e-03
                    -3.11762766926646229e-01 -9.28863195230936189e-01 3.52724892876299986e-03
                    -9.70548332676287262e-01 -1.42817000235277602e-01 3.93705547530200008e-03
                    -4.71015240470604279e-01 -8.63832466567013291e-01 3.16988483898999985e-03
                    -6.05620017736163474e-01 -7.76100660816042343e-01 2.76228288478600013e-03
                    -7.40289715445976160e-02 -9.90693239741754383e-01 2.49096311459699998e-03
                    -7.55715380409447968e-01 -6.45147335749373885e-01 2.36384376675300009e-03
                    -9.12089908253746806e-01 -3.94488725953920483e-01 2.34718358962999986e-03
                    -8.43257211940908857e-01 -5.26002453225995259e-01 2.35659053874399992e-03
                    -9.60608006066268838e-01 -2.57986117049701202e-01 1.99100374207599991e-03
                    -2.23329052372056358e-01 -9.70194673097433014e-01 1.82066169141500002e-03
                    -3.79684066987245328e-01 -9.21082693543939346e-01 1.58140143397599996e-03
                    -6.59642882473617354e-01 -7.46722135420716859e-01 1.35099198174799991e-03
                    -5.31373272478381842e-01 -8.44078282981892469e-01 1.18611055353199995e-03
                    -9.91979719359388223e-01 -1.14645109803836506e-01 8.60165845832999969e-04
                    ];

        end

    case 45

        xyw=[
            -0.00000000000000000e+00 0.00000000000000000e+00 2.02458760594159994e-02
            1.52210994096083990e-01 -0.00000000000000000e+00 1.99322866055859985e-02
            7.61153948653812418e-02 -1.31988508928929216e-01 1.99387928760580008e-02
            2.26636653348267103e-01 -1.30854571479858739e-01 1.93189446791990002e-02
            -2.70915098537134735e-14 -2.61977592661119973e-01 1.93368486413840017e-02
            3.00982456602689019e-01 -0.00000000000000000e+00 1.90055028194579985e-02
            1.50568625022072106e-01 -2.60830507191722349e-01 1.90306740276730002e-02
            3.72002746220695180e-01 -1.28665935003748372e-01 1.81067876939850006e-02
            2.97751796505546396e-01 -2.57466027919723617e-01 1.81271169802999996e-02
            7.44394989446407335e-02 -3.86852851302412093e-01 1.81511299764119984e-02
            4.42812603525170989e-01 -0.00000000000000000e+00 1.75030759965440000e-02
            2.21656067145843128e-01 -3.83431902265917279e-01 1.75615160507559993e-02
            4.38193981544557798e-01 -2.51983386755809891e-01 1.66687113692470003e-02
            -5.23135112139367856e-14 -5.05876852396955012e-01 1.67172862312580002e-02
            3.63951778066581133e-01 -3.76681382932519404e-01 1.64130085647620008e-02
            5.08746785450744365e-01 -1.25429971039809546e-01 1.63507786656450008e-02
            1.45417029773357825e-01 -5.03586521780531871e-01 1.64342756722259987e-02
            2.87653639027143193e-01 -4.96716175209229671e-01 1.55998901952370001e-02
            5.74210205413761998e-01 -0.00000000000000000e+00 1.54797288138779996e-02
            4.98292201667248413e-01 -3.66448855040012400e-01 1.47665979484150004e-02
            5.68682398622324503e-01 -2.44086712750831714e-01 1.47179465003099991e-02
            7.08714917211194506e-02 -6.14977272747694537e-01 1.47972093811449996e-02
            4.23809579053108398e-01 -4.85104489156514385e-01 1.42617949300500001e-02
            2.11101549862390298e-01 -6.09237702369623157e-01 1.42712596576130002e-02
            6.33565410390516948e-01 -1.20795080362137375e-01 1.41151188369639993e-02
            3.47006172882818764e-01 -5.97487710427306085e-01 1.32493821887789999e-02
            6.91997104509263994e-01 -0.00000000000000000e+00 1.30055367364900008e-02
            6.22185635749553545e-01 -3.51384034636510356e-01 1.27030906468969997e-02
            -7.39184834281166473e-14 -7.14799080827328015e-01 1.27423108036520006e-02
            5.51690305310132523e-01 -4.67685679155658118e-01 1.25042533015419996e-02
            1.36319430159902333e-01 -7.11377427247488314e-01 1.24986552094939994e-02
            6.86537994081489189e-01 -2.32196602285081738e-01 1.23542073640220008e-02
            4.76603757972521191e-01 -5.78639696900194012e-01 1.17927179330499995e-02
            2.69961360445662435e-01 -7.00878871062706410e-01 1.17916925161279992e-02
            7.43841000113884321e-01 -1.13365595045233608e-01 1.14857594062139998e-02
            3.99094221037978814e-01 -6.82057538303944222e-01 1.06628965622100006e-02
            7.94136569908392986e-01 -0.00000000000000000e+00 1.01966483512540007e-02
            6.69907398166725399e-01 -4.41438511096334696e-01 1.04164928844220003e-02
            6.55939651942629015e-02 -8.00441611318205681e-01 1.04279066439950004e-02
            7.33824366818358165e-01 -3.27974611220898826e-01 1.03071638422760001e-02
            5.99128429733798096e-01 -5.50168796114204661e-01 1.00103391109200007e-02
            1.95462681952353473e-01 -7.92377388153392648e-01 9.98527063465400044e-03
            7.89860710389377552e-01 -2.12667452681773367e-01 9.67280920408999934e-03
            5.23394657443880362e-01 -6.52152827307776017e-01 9.15769969163600026e-03
            3.22245581914873958e-01 -7.75222797235464389e-01 9.18480449543600079e-03
            8.38349121382036766e-01 -1.00693303295710371e-01 8.62284689718400069e-03
            4.46166696749369707e-01 -7.45742709939555937e-01 8.03784996659000044e-03
            -9.03846830395655001e-14 -8.74028867493999995e-01 8.34959134268399919e-03
            7.75607589274393328e-01 -4.03693505854362311e-01 7.96288216740500057e-03
            7.13123951646002174e-01 -5.08914248592499896e-01 7.90133825843000043e-03
            1.27675490592769392e-01 -8.69249778197293210e-01 8.10029314892100026e-03
            8.29453971327438699e-01 -2.94845243877633778e-01 7.54518598749299996e-03
            8.80749320822837967e-01 -0.00000000000000000e+00 7.35849825814199974e-03
            6.43152953439124042e-01 -6.08244890254437953e-01 7.41827359928100023e-03
            2.52429510441020188e-01 -8.54533998298993125e-01 7.47205044077199967e-03
            8.74572556408494184e-01 -1.88205624365368251e-01 6.54425846942000011e-03
            5.68077520983107820e-01 -7.00022431819166568e-01 6.51785994807899983e-03
            3.74590001829441577e-01 -8.27144056554009133e-01 6.64310052227299989e-03
            9.12274668830007962e-01 -9.81637218823833563e-02 5.27854944220499962e-03
            4.96548026313542279e-01 -7.82332915196957690e-01 5.60024111494999977e-03
            8.37476491170423443e-01 -4.08449697644291065e-01 6.19966254443300030e-03
            6.49325331691318064e-02 -9.29661657289040044e-01 6.31006618110400031e-03
            7.74255045579929857e-01 -5.18963235460558114e-01 6.17960595159099992e-03
            9.36258246766654034e-01 -0.00000000000000000e+00 4.07169039526200042e-03
            8.89153833818208228e-01 -2.93809913859517380e-01 5.64690298830199978e-03
            7.01589273271743807e-01 -6.20703053629688584e-01 5.63674464061400017e-03
            1.93033311584525469e-01 -9.17456241063337119e-01 5.84772160884900003e-03
            6.23768144105465550e-01 -7.10383380914756501e-01 4.56463969646399986e-03
            3.17413818842270656e-01 -8.91618644767587787e-01 5.11275598922100033e-03
            9.31312850992501828e-01 -1.88161098107404084e-01 4.57688070603100034e-03
            4.39067162358070029e-01 -8.48778013665778652e-01 4.35252839283799958e-03
            9.59920254721772781e-01 -7.40166005326032689e-02 3.69664490447900018e-03
            5.62791635456322381e-01 -7.83155669502999752e-01 3.80211601602800002e-03
            -1.00482309998767110e-13 -9.71673923699646003e-01 4.03054664752600014e-03
            8.45252589657203179e-01 -4.79571303698671014e-01 4.13560867754900023e-03
            9.02509168909957138e-01 -3.62710598056594113e-01 3.97254421476299965e-03
            7.74732119215064241e-01 -5.88638256436795348e-01 3.98032864307300022e-03
            1.26347399725497855e-01 -9.64791515712787984e-01 3.87681177408000016e-03
            6.91843706998913843e-01 -6.87593630030764125e-01 3.47671442779000014e-03
            2.51759131719660911e-01 -9.43324052526007928e-01 3.48213363548300009e-03
            9.46385640515023052e-01 -2.47611796223856123e-01 3.06415610903700009e-03
            3.75822932896636597e-01 -9.05166413020864535e-01 3.01572401684599986e-03
            4.98254677479377683e-01 -8.47552640180435524e-01 2.63043093963999993e-03
            9.74866919522095809e-01 -1.35122072276181671e-01 2.27317455869500019e-03
            9.84357949137810051e-01 -0.00000000000000000e+00 2.41453781182199984e-03
            6.20376294412741114e-01 -7.68087169927461844e-01 2.31708108782200002e-03
            8.87227844961913759e-01 -4.49432193511732136e-01 1.81699378494500009e-03
            5.44821645187927583e-02 -9.93150326521792359e-01 1.51565356696700009e-03
            8.20098236972424011e-01 -5.63094852116541311e-01 1.77989055310800006e-03
            7.37515269235189730e-01 -6.67925588083856270e-01 1.69236846262999993e-03
            1.69246533010959710e-01 -9.80575620695307015e-01 1.53239216352300009e-03
            9.38582387799713036e-01 -3.30741854888869868e-01 1.60648894496300009e-03
            2.91783145636952845e-01 -9.52053331312732176e-01 1.41898790055000005e-03
            4.16406914511321558e-01 -9.05227299095174853e-01 1.25118272175400009e-03
            9.75018245293556740e-01 -2.06524863155012228e-01 1.16165907201899992e-03
            5.36722916695014596e-01 -8.40087370723997751e-01 1.08911600163899992e-03
            9.94560436178089535e-01 -6.96924625355444111e-02 1.07131208582300001e-03
            6.49172702694666537e-01 -7.58361617637381857e-01 6.96201087704000012e-04
            7.61153948653812418e-02 1.31988508928929216e-01 1.99387928760580008e-02
            2.26636653348267103e-01 1.30854571479858739e-01 1.93189446791990002e-02
            -2.70915098537134735e-14 2.61977592661119973e-01 1.93368486413840017e-02
            1.50568625022072106e-01 2.60830507191722349e-01 1.90306740276730002e-02
            3.72002746220695180e-01 1.28665935003748372e-01 1.81067876939850006e-02
            2.97751796505546396e-01 2.57466027919723617e-01 1.81271169802999996e-02
            7.44394989446407335e-02 3.86852851302412093e-01 1.81511299764119984e-02
            2.21656067145843128e-01 3.83431902265917279e-01 1.75615160507559993e-02
            4.38193981544557798e-01 2.51983386755809891e-01 1.66687113692470003e-02
            -5.23135112139367856e-14 5.05876852396955012e-01 1.67172862312580002e-02
            3.63951778066581133e-01 3.76681382932519404e-01 1.64130085647620008e-02
            5.08746785450744365e-01 1.25429971039809546e-01 1.63507786656450008e-02
            1.45417029773357825e-01 5.03586521780531871e-01 1.64342756722259987e-02
            2.87653639027143193e-01 4.96716175209229671e-01 1.55998901952370001e-02
            4.98292201667248413e-01 3.66448855040012400e-01 1.47665979484150004e-02
            5.68682398622324503e-01 2.44086712750831714e-01 1.47179465003099991e-02
            7.08714917211194506e-02 6.14977272747694537e-01 1.47972093811449996e-02
            4.23809579053108398e-01 4.85104489156514385e-01 1.42617949300500001e-02
            2.11101549862390298e-01 6.09237702369623157e-01 1.42712596576130002e-02
            6.33565410390516948e-01 1.20795080362137375e-01 1.41151188369639993e-02
            3.47006172882818764e-01 5.97487710427306085e-01 1.32493821887789999e-02
            6.22185635749553545e-01 3.51384034636510356e-01 1.27030906468969997e-02
            -7.39184834281166473e-14 7.14799080827328015e-01 1.27423108036520006e-02
            5.51690305310132523e-01 4.67685679155658118e-01 1.25042533015419996e-02
            1.36319430159902333e-01 7.11377427247488314e-01 1.24986552094939994e-02
            6.86537994081489189e-01 2.32196602285081738e-01 1.23542073640220008e-02
            4.76603757972521191e-01 5.78639696900194012e-01 1.17927179330499995e-02
            2.69961360445662435e-01 7.00878871062706410e-01 1.17916925161279992e-02
            7.43841000113884321e-01 1.13365595045233608e-01 1.14857594062139998e-02
            3.99094221037978814e-01 6.82057538303944222e-01 1.06628965622100006e-02
            6.69907398166725399e-01 4.41438511096334696e-01 1.04164928844220003e-02
            6.55939651942629015e-02 8.00441611318205681e-01 1.04279066439950004e-02
            7.33824366818358165e-01 3.27974611220898826e-01 1.03071638422760001e-02
            5.99128429733798096e-01 5.50168796114204661e-01 1.00103391109200007e-02
            1.95462681952353473e-01 7.92377388153392648e-01 9.98527063465400044e-03
            7.89860710389377552e-01 2.12667452681773367e-01 9.67280920408999934e-03
            5.23394657443880362e-01 6.52152827307776017e-01 9.15769969163600026e-03
            3.22245581914873958e-01 7.75222797235464389e-01 9.18480449543600079e-03
            8.38349121382036766e-01 1.00693303295710371e-01 8.62284689718400069e-03
            4.46166696749369707e-01 7.45742709939555937e-01 8.03784996659000044e-03
            -9.03846830395655001e-14 8.74028867493999995e-01 8.34959134268399919e-03
            7.75607589274393328e-01 4.03693505854362311e-01 7.96288216740500057e-03
            7.13123951646002174e-01 5.08914248592499896e-01 7.90133825843000043e-03
            1.27675490592769392e-01 8.69249778197293210e-01 8.10029314892100026e-03
            8.29453971327438699e-01 2.94845243877633778e-01 7.54518598749299996e-03
            6.43152953439124042e-01 6.08244890254437953e-01 7.41827359928100023e-03
            2.52429510441020188e-01 8.54533998298993125e-01 7.47205044077199967e-03
            8.74572556408494184e-01 1.88205624365368251e-01 6.54425846942000011e-03
            5.68077520983107820e-01 7.00022431819166568e-01 6.51785994807899983e-03
            3.74590001829441577e-01 8.27144056554009133e-01 6.64310052227299989e-03
            9.12274668830007962e-01 9.81637218823833563e-02 5.27854944220499962e-03
            4.96548026313542279e-01 7.82332915196957690e-01 5.60024111494999977e-03
            8.37476491170423443e-01 4.08449697644291065e-01 6.19966254443300030e-03
            6.49325331691318064e-02 9.29661657289040044e-01 6.31006618110400031e-03
            7.74255045579929857e-01 5.18963235460558114e-01 6.17960595159099992e-03
            8.89153833818208228e-01 2.93809913859517380e-01 5.64690298830199978e-03
            7.01589273271743807e-01 6.20703053629688584e-01 5.63674464061400017e-03
            1.93033311584525469e-01 9.17456241063337119e-01 5.84772160884900003e-03
            6.23768144105465550e-01 7.10383380914756501e-01 4.56463969646399986e-03
            3.17413818842270656e-01 8.91618644767587787e-01 5.11275598922100033e-03
            9.31312850992501828e-01 1.88161098107404084e-01 4.57688070603100034e-03
            4.39067162358070029e-01 8.48778013665778652e-01 4.35252839283799958e-03
            9.59920254721772781e-01 7.40166005326032689e-02 3.69664490447900018e-03
            5.62791635456322381e-01 7.83155669502999752e-01 3.80211601602800002e-03
            -1.00482309998767110e-13 9.71673923699646003e-01 4.03054664752600014e-03
            8.45252589657203179e-01 4.79571303698671014e-01 4.13560867754900023e-03
            9.02509168909957138e-01 3.62710598056594113e-01 3.97254421476299965e-03
            7.74732119215064241e-01 5.88638256436795348e-01 3.98032864307300022e-03
            1.26347399725497855e-01 9.64791515712787984e-01 3.87681177408000016e-03
            6.91843706998913843e-01 6.87593630030764125e-01 3.47671442779000014e-03
            2.51759131719660911e-01 9.43324052526007928e-01 3.48213363548300009e-03
            9.46385640515023052e-01 2.47611796223856123e-01 3.06415610903700009e-03
            3.75822932896636597e-01 9.05166413020864535e-01 3.01572401684599986e-03
            4.98254677479377683e-01 8.47552640180435524e-01 2.63043093963999993e-03
            9.74866919522095809e-01 1.35122072276181671e-01 2.27317455869500019e-03
            6.20376294412741114e-01 7.68087169927461844e-01 2.31708108782200002e-03
            8.87227844961913759e-01 4.49432193511732136e-01 1.81699378494500009e-03
            5.44821645187927583e-02 9.93150326521792359e-01 1.51565356696700009e-03
            8.20098236972424011e-01 5.63094852116541311e-01 1.77989055310800006e-03
            7.37515269235189730e-01 6.67925588083856270e-01 1.69236846262999993e-03
            1.69246533010959710e-01 9.80575620695307015e-01 1.53239216352300009e-03
            9.38582387799713036e-01 3.30741854888869868e-01 1.60648894496300009e-03
            2.91783145636952845e-01 9.52053331312732176e-01 1.41898790055000005e-03
            4.16406914511321558e-01 9.05227299095174853e-01 1.25118272175400009e-03
            9.75018245293556740e-01 2.06524863155012228e-01 1.16165907201899992e-03
            5.36722916695014596e-01 8.40087370723997751e-01 1.08911600163899992e-03
            9.94560436178089535e-01 6.96924625355444111e-02 1.07131208582300001e-03
            6.49172702694666537e-01 7.58361617637381857e-01 6.96201087704000012e-04
            -1.52210994096083990e-01 1.86404706714805945e-17 1.99322866055859985e-02
            -7.61153948653812418e-02 1.31988508928929216e-01 1.99387928760580008e-02
            -2.26636653348267103e-01 1.30854571479858739e-01 1.93189446791990002e-02
            -3.00982456602689019e-01 3.68597202077990255e-17 1.90055028194579985e-02
            -1.50568625022072078e-01 2.60830507191722349e-01 1.90306740276730002e-02
            -3.72002746220695180e-01 1.28665935003748427e-01 1.81067876939850006e-02
            -2.97751796505546396e-01 2.57466027919723617e-01 1.81271169802999996e-02
            -7.44394989446406918e-02 3.86852851302412093e-01 1.81511299764119984e-02
            -4.42812603525170989e-01 5.42289037529206609e-17 1.75030759965440000e-02
            -2.21656067145842989e-01 3.83431902265917390e-01 1.75615160507559993e-02
            -4.38193981544557742e-01 2.51983386755809946e-01 1.66687113692470003e-02
            -3.63951778066581133e-01 3.76681382932519404e-01 1.64130085647620008e-02
            -5.08746785450744365e-01 1.25429971039809574e-01 1.63507786656450008e-02
            -1.45417029773357742e-01 5.03586521780531871e-01 1.64342756722259987e-02
            -2.87653639027143138e-01 4.96716175209229727e-01 1.55998901952370001e-02
            -5.74210205413761998e-01 7.03204690097707781e-17 1.54797288138779996e-02
            -4.98292201667248302e-01 3.66448855040012511e-01 1.47665979484150004e-02
            -5.68682398622324614e-01 2.44086712750831714e-01 1.47179465003099991e-02
            -7.08714917211193812e-02 6.14977272747694537e-01 1.47972093811449996e-02
            -4.23809579053108287e-01 4.85104489156514440e-01 1.42617949300500001e-02
            -2.11101549862390214e-01 6.09237702369623157e-01 1.42712596576130002e-02
            -6.33565410390516948e-01 1.20795080362137472e-01 1.41151188369639993e-02
            -3.47006172882818764e-01 5.97487710427306085e-01 1.32493821887789999e-02
            -6.91997104509263994e-01 8.47452039056506602e-17 1.30055367364900008e-02
            -6.22185635749553434e-01 3.51384034636510523e-01 1.27030906468969997e-02
            -5.51690305310132523e-01 4.67685679155658007e-01 1.25042533015419996e-02
            -1.36319430159902250e-01 7.11377427247488425e-01 1.24986552094939994e-02
            -6.86537994081489189e-01 2.32196602285081932e-01 1.23542073640220008e-02
            -4.76603757972521191e-01 5.78639696900194012e-01 1.17927179330499995e-02
            -2.69961360445662324e-01 7.00878871062706410e-01 1.17916925161279992e-02
            -7.43841000113884321e-01 1.13365595045233622e-01 1.14857594062139998e-02
            -3.99094221037978869e-01 6.82057538303944222e-01 1.06628965622100006e-02
            -7.94136569908392986e-01 9.72536808424171707e-17 1.01966483512540007e-02
            -6.69907398166725288e-01 4.41438511096334862e-01 1.04164928844220003e-02
            -6.55939651942628044e-02 8.00441611318205681e-01 1.04279066439950004e-02
            -7.33824366818358165e-01 3.27974611220898937e-01 1.03071638422760001e-02
            -5.99128429733798096e-01 5.50168796114204550e-01 1.00103391109200007e-02
            -1.95462681952353390e-01 7.92377388153392648e-01 9.98527063465400044e-03
            -7.89860710389377552e-01 2.12667452681773589e-01 9.67280920408999934e-03
            -5.23394657443880251e-01 6.52152827307776239e-01 9.15769969163600026e-03
            -3.22245581914873847e-01 7.75222797235464389e-01 9.18480449543600079e-03
            -8.38349121382036766e-01 1.00693303295710565e-01 8.62284689718400069e-03
            -4.46166696749369651e-01 7.45742709939556048e-01 8.03784996659000044e-03
            -7.75607589274393217e-01 4.03693505854362478e-01 7.96288216740500057e-03
            -7.13123951646002063e-01 5.08914248592500007e-01 7.90133825843000043e-03
            -1.27675490592769308e-01 8.69249778197293210e-01 8.10029314892100026e-03
            -8.29453971327438699e-01 2.94845243877633667e-01 7.54518598749299996e-03
            -8.80749320822837967e-01 1.07860683659689383e-16 7.35849825814199974e-03
            -6.43152953439123820e-01 6.08244890254438064e-01 7.41827359928100023e-03
            -2.52429510441020077e-01 8.54533998298993125e-01 7.47205044077199967e-03
            -8.74572556408494184e-01 1.88205624365368279e-01 6.54425846942000011e-03
            -5.68077520983107598e-01 7.00022431819166790e-01 6.51785994807899983e-03
            -3.74590001829441466e-01 8.27144056554009133e-01 6.64310052227299989e-03
            -9.12274668830007851e-01 9.81637218823835506e-02 5.27854944220499962e-03
            -4.96548026313542390e-01 7.82332915196957690e-01 5.60024111494999977e-03
            -8.37476491170423443e-01 4.08449697644291010e-01 6.19966254443300030e-03
            -6.49325331691316815e-02 9.29661657289040044e-01 6.31006618110400031e-03
            -7.74255045579929857e-01 5.18963235460558003e-01 6.17960595159099992e-03
            -9.36258246766654034e-01 1.14658566507809565e-16 4.07169039526200042e-03
            -8.89153833818208228e-01 2.93809913859517491e-01 5.64690298830199978e-03
            -7.01589273271743807e-01 6.20703053629688695e-01 5.63674464061400017e-03
            -1.93033311584525358e-01 9.17456241063337119e-01 5.84772160884900003e-03
            -6.23768144105465550e-01 7.10383380914756501e-01 4.56463969646399986e-03
            -3.17413818842270601e-01 8.91618644767587787e-01 5.11275598922100033e-03
            -9.31312850992501828e-01 1.88161098107404196e-01 4.57688070603100034e-03
            -4.39067162358069751e-01 8.48778013665778874e-01 4.35252839283799958e-03
            -9.59920254721772781e-01 7.40166005326035326e-02 3.69664490447900018e-03
            -5.62791635456322159e-01 7.83155669502999863e-01 3.80211601602800002e-03
            -8.45252589657203068e-01 4.79571303698671125e-01 4.13560867754900023e-03
            -9.02509168909957027e-01 3.62710598056594336e-01 3.97254421476299965e-03
            -7.74732119215064019e-01 5.88638256436795571e-01 3.98032864307300022e-03
            -1.26347399725497744e-01 9.64791515712787984e-01 3.87681177408000016e-03
            -6.91843706998913843e-01 6.87593630030764125e-01 3.47671442779000014e-03
            -2.51759131719660800e-01 9.43324052526007928e-01 3.48213363548300009e-03
            -9.46385640515022941e-01 2.47611796223856373e-01 3.06415610903700009e-03
            -3.75822932896636486e-01 9.05166413020864646e-01 3.01572401684599986e-03
            -4.98254677479377572e-01 8.47552640180435635e-01 2.63043093963999993e-03
            -9.74866919522095809e-01 1.35122072276181865e-01 2.27317455869500019e-03
            -9.84357949137810051e-01 1.20549081162687223e-16 2.41453781182199984e-03
            -6.20376294412740781e-01 7.68087169927461955e-01 2.31708108782200002e-03
            -8.87227844961913759e-01 4.49432193511732081e-01 1.81699378494500009e-03
            -5.44821645187926404e-02 9.93150326521792359e-01 1.51565356696700009e-03
            -8.20098236972424011e-01 5.63094852116541311e-01 1.77989055310800006e-03
            -7.37515269235189619e-01 6.67925588083856492e-01 1.69236846262999993e-03
            -1.69246533010959571e-01 9.80575620695307015e-01 1.53239216352300009e-03
            -9.38582387799712925e-01 3.30741854888870146e-01 1.60648894496300009e-03
            -2.91783145636952734e-01 9.52053331312732176e-01 1.41898790055000005e-03
            -4.16406914511321447e-01 9.05227299095174964e-01 1.25118272175400009e-03
            -9.75018245293556740e-01 2.06524863155012145e-01 1.16165907201899992e-03
            -5.36722916695014485e-01 8.40087370723997751e-01 1.08911600163899992e-03
            -9.94560436178089535e-01 6.96924625355445360e-02 1.07131208582300001e-03
            -6.49172702694666426e-01 7.58361617637381857e-01 6.96201087704000012e-04
            -7.61153948653812834e-02 -1.31988508928929216e-01 1.99387928760580008e-02
            -2.26636653348267159e-01 -1.30854571479858683e-01 1.93189446791990002e-02
            -1.50568625022072161e-01 -2.60830507191722349e-01 1.90306740276730002e-02
            -3.72002746220695180e-01 -1.28665935003748344e-01 1.81067876939850006e-02
            -2.97751796505546451e-01 -2.57466027919723561e-01 1.81271169802999996e-02
            -7.44394989446407057e-02 -3.86852851302412093e-01 1.81511299764119984e-02
            -2.21656067145843072e-01 -3.83431902265917335e-01 1.75615160507559993e-02
            -4.38193981544557798e-01 -2.51983386755809835e-01 1.66687113692470003e-02
            -3.63951778066581244e-01 -3.76681382932519349e-01 1.64130085647620008e-02
            -5.08746785450744476e-01 -1.25429971039809435e-01 1.63507786656450008e-02
            -1.45417029773357881e-01 -5.03586521780531871e-01 1.64342756722259987e-02
            -2.87653639027143249e-01 -4.96716175209229671e-01 1.55998901952370001e-02
            -4.98292201667248413e-01 -3.66448855040012400e-01 1.47665979484150004e-02
            -5.68682398622324614e-01 -2.44086712750831575e-01 1.47179465003099991e-02
            -7.08714917211196727e-02 -6.14977272747694537e-01 1.47972093811449996e-02
            -4.23809579053108454e-01 -4.85104489156514385e-01 1.42617949300500001e-02
            -2.11101549862390464e-01 -6.09237702369622935e-01 1.42712596576130002e-02
            -6.33565410390516948e-01 -1.20795080362137319e-01 1.41151188369639993e-02
            -3.47006172882818931e-01 -5.97487710427305974e-01 1.32493821887789999e-02
            -6.22185635749553545e-01 -3.51384034636510356e-01 1.27030906468969997e-02
            -5.51690305310132745e-01 -4.67685679155657896e-01 1.25042533015419996e-02
            -1.36319430159902416e-01 -7.11377427247488314e-01 1.24986552094939994e-02
            -6.86537994081489189e-01 -2.32196602285081766e-01 1.23542073640220008e-02
            -4.76603757972521025e-01 -5.78639696900194123e-01 1.17927179330499995e-02
            -2.69961360445662824e-01 -7.00878871062706299e-01 1.17916925161279992e-02
            -7.43841000113884432e-01 -1.13365595045233442e-01 1.14857594062139998e-02
            -3.99094221037979036e-01 -6.82057538303944111e-01 1.06628965622100006e-02
            -6.69907398166725399e-01 -4.41438511096334696e-01 1.04164928844220003e-02
            -6.55939651942628182e-02 -8.00441611318205681e-01 1.04279066439950004e-02
            -7.33824366818358276e-01 -3.27974611220898771e-01 1.03071638422760001e-02
            -5.99128429733798207e-01 -5.50168796114204328e-01 1.00103391109200007e-02
            -1.95462681952353751e-01 -7.92377388153392537e-01 9.98527063465400044e-03
            -7.89860710389377552e-01 -2.12667452681773367e-01 9.67280920408999934e-03
            -5.23394657443880695e-01 -6.52152827307775795e-01 9.15769969163600026e-03
            -3.22245581914873735e-01 -7.75222797235464500e-01 9.18480449543600079e-03
            -8.38349121382036766e-01 -1.00693303295710357e-01 8.62284689718400069e-03
            -4.46166696749369818e-01 -7.45742709939555937e-01 8.03784996659000044e-03
            -7.75607589274393328e-01 -4.03693505854362256e-01 7.96288216740500057e-03
            -7.13123951646002285e-01 -5.08914248592499785e-01 7.90133825843000043e-03
            -1.27675490592769336e-01 -8.69249778197293210e-01 8.10029314892100026e-03
            -8.29453971327438810e-01 -2.94845243877633445e-01 7.54518598749299996e-03
            -6.43152953439124042e-01 -6.08244890254437953e-01 7.41827359928100023e-03
            -2.52429510441020077e-01 -8.54533998298993125e-01 7.47205044077199967e-03
            -8.74572556408494184e-01 -1.88205624365368057e-01 6.54425846942000011e-03
            -5.68077520983107820e-01 -7.00022431819166679e-01 6.51785994807899983e-03
            -3.74590001829441854e-01 -8.27144056554009022e-01 6.64310052227299989e-03
            -9.12274668830007962e-01 -9.81637218823833146e-02 5.27854944220499962e-03
            -4.96548026313542612e-01 -7.82332915196957468e-01 5.60024111494999977e-03
            -8.37476491170423554e-01 -4.08449697644290843e-01 6.19966254443300030e-03
            -6.49325331691321256e-02 -9.29661657289039933e-01 6.31006618110400031e-03
            -7.74255045579929968e-01 -5.18963235460557781e-01 6.17960595159099992e-03
            -8.89153833818208339e-01 -2.93809913859517269e-01 5.64690298830199978e-03
            -7.01589273271743918e-01 -6.20703053629688473e-01 5.63674464061400017e-03
            -1.93033311584525996e-01 -9.17456241063337008e-01 5.84772160884900003e-03
            -6.23768144105465661e-01 -7.10383380914756390e-01 4.56463969646399986e-03
            -3.17413818842270656e-01 -8.91618644767587787e-01 5.11275598922100033e-03
            -9.31312850992501939e-01 -1.88161098107403946e-01 4.57688070603100034e-03
            -4.39067162358069973e-01 -8.48778013665778763e-01 4.35252839283799958e-03
            -9.59920254721772781e-01 -7.40166005326032966e-02 3.69664490447900018e-03
            -5.62791635456322381e-01 -7.83155669502999641e-01 3.80211601602800002e-03
            -8.45252589657203179e-01 -4.79571303698670903e-01 4.13560867754900023e-03
            -9.02509168909957138e-01 -3.62710598056594113e-01 3.97254421476299965e-03
            -7.74732119215064241e-01 -5.88638256436795348e-01 3.98032864307300022e-03
            -1.26347399725497994e-01 -9.64791515712787984e-01 3.87681177408000016e-03
            -6.91843706998914065e-01 -6.87593630030764014e-01 3.47671442779000014e-03
            -2.51759131719661411e-01 -9.43324052526007706e-01 3.48213363548300009e-03
            -9.46385640515023052e-01 -2.47611796223856123e-01 3.06415610903700009e-03
            -3.75822932896636708e-01 -9.05166413020864535e-01 3.01572401684599986e-03
            -4.98254677479377683e-01 -8.47552640180435413e-01 2.63043093963999993e-03
            -9.74866919522095809e-01 -1.35122072276181615e-01 2.27317455869500019e-03
            -6.20376294412741003e-01 -7.68087169927461844e-01 2.31708108782200002e-03
            -8.87227844961913870e-01 -4.49432193511731859e-01 1.81699378494500009e-03
            -5.44821645187926612e-02 -9.93150326521792359e-01 1.51565356696700009e-03
            -8.20098236972424122e-01 -5.63094852116541089e-01 1.77989055310800006e-03
            -7.37515269235189730e-01 -6.67925588083856270e-01 1.69236846262999993e-03
            -1.69246533010960043e-01 -9.80575620695307015e-01 1.53239216352300009e-03
            -9.38582387799713036e-01 -3.30741854888869924e-01 1.60648894496300009e-03
            -2.91783145636953400e-01 -9.52053331312731954e-01 1.41898790055000005e-03
            -4.16406914511321669e-01 -9.05227299095174853e-01 1.25118272175400009e-03
            -9.75018245293556851e-01 -2.06524863155011895e-01 1.16165907201899992e-03
            -5.36722916695014374e-01 -8.40087370723997862e-01 1.08911600163899992e-03
            -9.94560436178089535e-01 -6.96924625355443000e-02 1.07131208582300001e-03
            -6.49172702694666648e-01 -7.58361617637381746e-01 6.96201087704000012e-04
            ];


    case 49

        xyw=[
            -0.00000000000000000e+00 0.00000000000000000e+00 1.72152728643839986e-02
            1.40162001046804013e-01 -0.00000000000000000e+00 1.69800939929319987e-02
            7.00959649866139956e-02 -1.22013963230467842e-01 1.69963608751830010e-02
            2.09085990408722944e-01 -1.20758856025446809e-01 1.65293417033620001e-02
            -2.50816247683255716e-14 -2.42541804141446993e-01 1.65815168287819989e-02
            2.77915296893500974e-01 -0.00000000000000000e+00 1.62877015644839994e-02
            1.39092912392518281e-01 -2.41267231297798368e-01 1.63464538735140011e-02
            2.75753624248552487e-01 -2.37572881027699662e-01 1.56536808532279983e-02
            3.44426178957883411e-01 -1.18386788174632246e-01 1.56209624051599997e-02
            6.90542086154238982e-02 -3.58839697664831980e-01 1.57529835885969992e-02
            2.05938975761480042e-01 -3.55042554837130264e-01 1.52859058974919998e-02
            4.10759240703882977e-01 -0.00000000000000000e+00 1.51760987099460003e-02
            4.07413459425605695e-01 -2.31952747352193511e-01 1.45391577748200003e-02
            -4.86845755781260581e-14 -4.70784684152167010e-01 1.47774024859079998e-02
            3.39061499616248152e-01 -3.47894013676684544e-01 1.43739422110259994e-02
            4.73476035944830342e-01 -1.15289529048016939e-01 1.43190080035430004e-02
            1.36112001834573421e-01 -4.68200229648335131e-01 1.45429014476019993e-02
            2.69695952756031365e-01 -4.60737849124308707e-01 1.38491993647810001e-02
            5.35876124992857972e-01 0.00000000000000000e+00 1.36990486693940004e-02
            4.65571173408012018e-01 -3.38570570114858749e-01 1.30599870310389998e-02
            5.31141040575525469e-01 -2.25574144230106494e-01 1.30613613115730005e-02
            6.72417531977656802e-02 -5.74440508138183925e-01 1.34614859364179992e-02
            3.97956031410485578e-01 -4.49424057158134993e-01 1.27215172670719998e-02
            2.00448439189885252e-01 -5.68008713815135624e-01 1.29993267975569999e-02
            5.93093802694955041e-01 -1.12268509614250903e-01 1.26842903194569999e-02
            3.29672413780715601e-01 -5.55928518197362664e-01 1.20887136126920005e-02
            6.49777783581100055e-01 -0.00000000000000000e+00 1.19178629665440006e-02
            5.81891504668867632e-01 -3.29742771775893950e-01 1.14262134394630000e-02
            -6.94079108119212802e-14 -6.71181395364438993e-01 1.20800944981479998e-02
            5.17334353214284337e-01 -4.36775923227416152e-01 1.12022452541259993e-02
            6.43225850131183940e-01 -2.20685923529787720e-01 1.13116950123369996e-02
            1.32108484141322369e-01 -6.67226611391283475e-01 1.18542238867049999e-02
            4.51506633325892304e-01 -5.40330996613102799e-01 1.07424198339379998e-02
            2.61596582514102283e-01 -6.55784353180901225e-01 1.11824663963750008e-02
            6.99348217842765574e-01 -1.10582022959264406e-01 1.08065485960600009e-02
            3.85413206064164127e-01 -6.38487336212394840e-01 1.00777754150390003e-02
            7.48409155164684980e-01 0.00000000000000000e+00 9.94970923213200041e-03
            6.22546872015464436e-01 -4.28708611861134736e-01 9.46846143336499933e-03
            6.83335846938037617e-01 -3.26076854336383504e-01 9.69271546586900068e-03
            6.51137490201267183e-02 -7.55917110647262303e-01 1.04304334004890007e-02
            5.60233753993043093e-01 -5.27025882085813246e-01 8.99253670308900084e-03
            7.39491650734243611e-01 -2.19794840211143122e-01 9.48739923846200067e-03
            1.94006373119737702e-01 -7.46712952846628086e-01 9.99431725244900003e-03
            4.98507942383859592e-01 -6.20290246421132285e-01 8.53996870122200082e-03
            3.18637769847352759e-01 -7.29363550790668613e-01 9.11940264582200046e-03
            7.88670089677898312e-01 -1.10644612887255245e-01 8.87521055623300033e-03
            8.29348135045060997e-01 -0.00000000000000000e+00 7.99030134096199995e-03
            7.10305225499725079e-01 -4.28481870967066802e-01 8.05074749614799988e-03
            4.34456125396177506e-01 -7.07721657126255899e-01 7.78459653634299992e-03
            -8.60024640255488548e-14 -8.31652374120061988e-01 8.80266753628199984e-03
            6.48737320285741048e-01 -5.24615490016243080e-01 7.39990007382800019e-03
            7.68619846181405775e-01 -3.25791952380782623e-01 8.13334299876299939e-03
            1.28090154189880484e-01 -8.26262739400246415e-01 8.60000475803800006e-03
            8.19536605836078902e-01 -2.18023103014843289e-01 7.71027751666599967e-03
            2.53414327379380078e-01 -8.10569752993718362e-01 7.98250281608500062e-03
            5.91281919080588025e-01 -6.12984374028710532e-01 6.80887815498300037e-03
            8.61399635652120743e-01 -1.07538029444819785e-01 6.95469195100400020e-03
            3.71965010291021969e-01 -7.87137715992151876e-01 6.90081324771099971e-03
            5.31113707155597914e-01 -6.93708962570089627e-01 6.08886552220200012e-03
            7.20202177923419784e-01 -5.27315480709407924e-01 6.42604635464300022e-03
            7.84396447206010317e-01 -4.26844602778035487e-01 6.76111929125100021e-03
            6.36370689277424295e-02 -8.92150826573859690e-01 7.01676086174300023e-03
            8.94465956289006048e-01 0.00000000000000000e+00 6.01388939749199995e-03
            8.39751369848549034e-01 -3.19759394235543559e-01 6.47718814993200002e-03
            1.89682853842055327e-01 -8.79917616745109044e-01 6.66166852721499989e-03
            4.73550397830834802e-01 -7.67029127592730120e-01 5.37527515541099971e-03
            6.58111851167667350e-01 -6.21569854603836958e-01 5.86825802864900017e-03
            8.84259313777354894e-01 -2.09142904866452067e-01 5.82014553803700003e-03
            3.11279440534071417e-01 -8.56760841607447921e-01 5.90210411918599964e-03
            5.92109681606746818e-01 -7.06686865853840551e-01 5.05468425581599972e-03
            9.18643348946508898e-01 -1.01372037057271172e-01 4.96098634907999984e-03
            4.19343957925062294e-01 -8.31025602200504809e-01 4.53786299893900024e-03
            7.69384125047385292e-01 -5.40958707674880412e-01 4.63278203590699982e-03
            -9.74351682784861490e-14 -9.42207760437171027e-01 5.24723779192599969e-03
            8.27976504829669779e-01 -4.49799092322439753e-01 4.38885090972099962e-03
            1.26274395717196536e-01 -9.35289724407231127e-01 5.10983025173800034e-03
            9.44058125619724020e-01 -0.00000000000000000e+00 4.11252465550999979e-03
            8.73944275221654299e-01 -3.57990440926378162e-01 4.11305156997800007e-03
            5.32231982057495223e-01 -7.80733099630829597e-01 4.15891740887000018e-03
            7.08354641217884717e-01 -6.30786333530879206e-01 4.14019006363500024e-03
            2.50270052934766785e-01 -9.14999958338325481e-01 4.67968681326700036e-03
            9.12753181626832499e-01 -2.59510477251467031e-01 4.04996491839500032e-03
            9.44506998707056344e-01 -1.55997949457722662e-01 3.53989175112300004e-03
            3.65963019870668316e-01 -8.85544532448380606e-01 3.73576961318099979e-03
            6.43582590393861231e-01 -7.10642611351690046e-01 3.42295233433300003e-03
            4.78978173245638239e-01 -8.40645262192400211e-01 3.16317387453199979e-03
            9.67702272601057123e-01 -6.25224173268098260e-02 2.65823741461599988e-03
            5.89697586066004975e-01 -7.76609310273313591e-01 2.67617892018900006e-03
            8.19917784356321233e-01 -5.28181902224525324e-01 2.70279797596600013e-03
            6.25582079264620367e-02 -9.74488040178466686e-01 3.32356290236999986e-03
            8.71294574610500194e-01 -4.41925297246777704e-01 2.71758706882600006e-03
            7.64875803409654109e-01 -6.07977409475020547e-01 2.55934909111300022e-03
            9.14816945191667008e-01 -3.44367015105893515e-01 2.80761613499500018e-03
            1.86663214982910414e-01 -9.59766326541879766e-01 3.13674349357900015e-03
            9.51580213576980749e-01 -2.36314820624810129e-01 2.60715986157099983e-03
            3.06052920318043986e-01 -9.31974305341974674e-01 2.66902374090400001e-03
            7.00477868969778150e-01 -6.89249210565360038e-01 2.34081029261299983e-03
            4.21337094243707466e-01 -8.90449487847489007e-01 2.15399144859500019e-03
            9.77931349120474414e-01 -1.29153676091683572e-01 1.88963648898100010e-03
            9.87862253555504055e-01 -0.00000000000000000e+00 1.81726874741800003e-03
            5.34151205565022158e-01 -8.32524912275056428e-01 1.74199341019399992e-03
            6.40293442886031405e-01 -7.57551099039111109e-01 1.32907094839699994e-03
            8.42644153128901618e-01 -5.29785435489457446e-01 1.41949133903299996e-03
            -1.02946732516904113e-13 -9.95505134366292954e-01 1.43471623930399995e-03
            1.23927397776883996e-01 -9.87850182636107688e-01 1.39942996432999996e-03
            7.74838581919999769e-01 -6.25197959050785457e-01 1.28816548380300002e-03
            9.00562854921806100e-01 -4.24677346474962580e-01 1.33942398695400004e-03
            9.45240136607224835e-01 -3.13244549623881763e-01 1.28981769243700006e-03
            2.45183344909571665e-01 -9.65393646502639591e-01 1.25297072312499994e-03
            3.62818218907764556e-01 -9.28395466735493557e-01 1.03110146672799998e-03
            9.77250751076047086e-01 -1.97502329192228793e-01 9.64757667509000012e-04
            9.95287566748883301e-01 -6.81285415252920329e-02 8.26725988317999973e-04
            4.77242103183890987e-01 -8.76088303347900554e-01 7.88848272361000052e-04
            7.01958362179900153e-01 -7.09375701626268285e-01 6.75013383016000016e-04
            5.90879883000531514e-01 -8.04958502622752592e-01 5.68376940729000009e-04
            7.00959649866139956e-02 1.22013963230467842e-01 1.69963608751830010e-02
            2.09085990408722944e-01 1.20758856025446809e-01 1.65293417033620001e-02
            -2.50816247683255716e-14 2.42541804141446993e-01 1.65815168287819989e-02
            1.39092912392518281e-01 2.41267231297798368e-01 1.63464538735140011e-02
            2.75753624248552487e-01 2.37572881027699662e-01 1.56536808532279983e-02
            3.44426178957883411e-01 1.18386788174632246e-01 1.56209624051599997e-02
            6.90542086154238982e-02 3.58839697664831980e-01 1.57529835885969992e-02
            2.05938975761480042e-01 3.55042554837130264e-01 1.52859058974919998e-02
            4.07413459425605695e-01 2.31952747352193511e-01 1.45391577748200003e-02
            -4.86845755781260581e-14 4.70784684152167010e-01 1.47774024859079998e-02
            3.39061499616248152e-01 3.47894013676684544e-01 1.43739422110259994e-02
            4.73476035944830342e-01 1.15289529048016939e-01 1.43190080035430004e-02
            1.36112001834573421e-01 4.68200229648335131e-01 1.45429014476019993e-02
            2.69695952756031365e-01 4.60737849124308707e-01 1.38491993647810001e-02
            4.65571173408012018e-01 3.38570570114858749e-01 1.30599870310389998e-02
            5.31141040575525469e-01 2.25574144230106494e-01 1.30613613115730005e-02
            6.72417531977656802e-02 5.74440508138183925e-01 1.34614859364179992e-02
            3.97956031410485578e-01 4.49424057158134993e-01 1.27215172670719998e-02
            2.00448439189885252e-01 5.68008713815135624e-01 1.29993267975569999e-02
            5.93093802694955041e-01 1.12268509614250903e-01 1.26842903194569999e-02
            3.29672413780715601e-01 5.55928518197362664e-01 1.20887136126920005e-02
            5.81891504668867632e-01 3.29742771775893950e-01 1.14262134394630000e-02
            -6.94079108119212802e-14 6.71181395364438993e-01 1.20800944981479998e-02
            5.17334353214284337e-01 4.36775923227416152e-01 1.12022452541259993e-02
            6.43225850131183940e-01 2.20685923529787720e-01 1.13116950123369996e-02
            1.32108484141322369e-01 6.67226611391283475e-01 1.18542238867049999e-02
            4.51506633325892304e-01 5.40330996613102799e-01 1.07424198339379998e-02
            2.61596582514102283e-01 6.55784353180901225e-01 1.11824663963750008e-02
            6.99348217842765574e-01 1.10582022959264406e-01 1.08065485960600009e-02
            3.85413206064164127e-01 6.38487336212394840e-01 1.00777754150390003e-02
            6.22546872015464436e-01 4.28708611861134736e-01 9.46846143336499933e-03
            6.83335846938037617e-01 3.26076854336383504e-01 9.69271546586900068e-03
            6.51137490201267183e-02 7.55917110647262303e-01 1.04304334004890007e-02
            5.60233753993043093e-01 5.27025882085813246e-01 8.99253670308900084e-03
            7.39491650734243611e-01 2.19794840211143122e-01 9.48739923846200067e-03
            1.94006373119737702e-01 7.46712952846628086e-01 9.99431725244900003e-03
            4.98507942383859592e-01 6.20290246421132285e-01 8.53996870122200082e-03
            3.18637769847352759e-01 7.29363550790668613e-01 9.11940264582200046e-03
            7.88670089677898312e-01 1.10644612887255245e-01 8.87521055623300033e-03
            7.10305225499725079e-01 4.28481870967066802e-01 8.05074749614799988e-03
            4.34456125396177506e-01 7.07721657126255899e-01 7.78459653634299992e-03
            -8.60024640255488548e-14 8.31652374120061988e-01 8.80266753628199984e-03
            6.48737320285741048e-01 5.24615490016243080e-01 7.39990007382800019e-03
            7.68619846181405775e-01 3.25791952380782623e-01 8.13334299876299939e-03
            1.28090154189880484e-01 8.26262739400246415e-01 8.60000475803800006e-03
            8.19536605836078902e-01 2.18023103014843289e-01 7.71027751666599967e-03
            2.53414327379380078e-01 8.10569752993718362e-01 7.98250281608500062e-03
            5.91281919080588025e-01 6.12984374028710532e-01 6.80887815498300037e-03
            8.61399635652120743e-01 1.07538029444819785e-01 6.95469195100400020e-03
            3.71965010291021969e-01 7.87137715992151876e-01 6.90081324771099971e-03
            5.31113707155597914e-01 6.93708962570089627e-01 6.08886552220200012e-03
            7.20202177923419784e-01 5.27315480709407924e-01 6.42604635464300022e-03
            7.84396447206010317e-01 4.26844602778035487e-01 6.76111929125100021e-03
            6.36370689277424295e-02 8.92150826573859690e-01 7.01676086174300023e-03
            8.39751369848549034e-01 3.19759394235543559e-01 6.47718814993200002e-03
            1.89682853842055327e-01 8.79917616745109044e-01 6.66166852721499989e-03
            4.73550397830834802e-01 7.67029127592730120e-01 5.37527515541099971e-03
            6.58111851167667350e-01 6.21569854603836958e-01 5.86825802864900017e-03
            8.84259313777354894e-01 2.09142904866452067e-01 5.82014553803700003e-03
            3.11279440534071417e-01 8.56760841607447921e-01 5.90210411918599964e-03
            5.92109681606746818e-01 7.06686865853840551e-01 5.05468425581599972e-03
            9.18643348946508898e-01 1.01372037057271172e-01 4.96098634907999984e-03
            4.19343957925062294e-01 8.31025602200504809e-01 4.53786299893900024e-03
            7.69384125047385292e-01 5.40958707674880412e-01 4.63278203590699982e-03
            -9.74351682784861490e-14 9.42207760437171027e-01 5.24723779192599969e-03
            8.27976504829669779e-01 4.49799092322439753e-01 4.38885090972099962e-03
            1.26274395717196536e-01 9.35289724407231127e-01 5.10983025173800034e-03
            8.73944275221654299e-01 3.57990440926378162e-01 4.11305156997800007e-03
            5.32231982057495223e-01 7.80733099630829597e-01 4.15891740887000018e-03
            7.08354641217884717e-01 6.30786333530879206e-01 4.14019006363500024e-03
            2.50270052934766785e-01 9.14999958338325481e-01 4.67968681326700036e-03
            9.12753181626832499e-01 2.59510477251467031e-01 4.04996491839500032e-03
            9.44506998707056344e-01 1.55997949457722662e-01 3.53989175112300004e-03
            3.65963019870668316e-01 8.85544532448380606e-01 3.73576961318099979e-03
            6.43582590393861231e-01 7.10642611351690046e-01 3.42295233433300003e-03
            4.78978173245638239e-01 8.40645262192400211e-01 3.16317387453199979e-03
            9.67702272601057123e-01 6.25224173268098260e-02 2.65823741461599988e-03
            5.89697586066004975e-01 7.76609310273313591e-01 2.67617892018900006e-03
            8.19917784356321233e-01 5.28181902224525324e-01 2.70279797596600013e-03
            6.25582079264620367e-02 9.74488040178466686e-01 3.32356290236999986e-03
            8.71294574610500194e-01 4.41925297246777704e-01 2.71758706882600006e-03
            7.64875803409654109e-01 6.07977409475020547e-01 2.55934909111300022e-03
            9.14816945191667008e-01 3.44367015105893515e-01 2.80761613499500018e-03
            1.86663214982910414e-01 9.59766326541879766e-01 3.13674349357900015e-03
            9.51580213576980749e-01 2.36314820624810129e-01 2.60715986157099983e-03
            3.06052920318043986e-01 9.31974305341974674e-01 2.66902374090400001e-03
            7.00477868969778150e-01 6.89249210565360038e-01 2.34081029261299983e-03
            4.21337094243707466e-01 8.90449487847489007e-01 2.15399144859500019e-03
            9.77931349120474414e-01 1.29153676091683572e-01 1.88963648898100010e-03
            5.34151205565022158e-01 8.32524912275056428e-01 1.74199341019399992e-03
            6.40293442886031405e-01 7.57551099039111109e-01 1.32907094839699994e-03
            8.42644153128901618e-01 5.29785435489457446e-01 1.41949133903299996e-03
            -1.02946732516904113e-13 9.95505134366292954e-01 1.43471623930399995e-03
            1.23927397776883996e-01 9.87850182636107688e-01 1.39942996432999996e-03
            7.74838581919999769e-01 6.25197959050785457e-01 1.28816548380300002e-03
            9.00562854921806100e-01 4.24677346474962580e-01 1.33942398695400004e-03
            9.45240136607224835e-01 3.13244549623881763e-01 1.28981769243700006e-03
            2.45183344909571665e-01 9.65393646502639591e-01 1.25297072312499994e-03
            3.62818218907764556e-01 9.28395466735493557e-01 1.03110146672799998e-03
            9.77250751076047086e-01 1.97502329192228793e-01 9.64757667509000012e-04
            9.95287566748883301e-01 6.81285415252920329e-02 8.26725988317999973e-04
            4.77242103183890987e-01 8.76088303347900554e-01 7.88848272361000052e-04
            7.01958362179900153e-01 7.09375701626268285e-01 6.75013383016000016e-04
            5.90879883000531514e-01 8.04958502622752592e-01 5.68376940729000009e-04
            -1.40162001046804013e-01 1.71648945944056506e-17 1.69800939929319987e-02
            -7.00959649866139817e-02 1.22013963230467856e-01 1.69963608751830010e-02
            -2.09085990408722916e-01 1.20758856025446837e-01 1.65293417033620001e-02
            -2.77915296893500974e-01 3.40348078774712300e-17 1.62877015644839994e-02
            -1.39092912392518253e-01 2.41267231297798396e-01 1.63464538735140011e-02
            -2.75753624248552487e-01 2.37572881027699689e-01 1.56536808532279983e-02
            -3.44426178957883355e-01 1.18386788174632357e-01 1.56209624051599997e-02
            -6.90542086154238566e-02 3.58839697664831980e-01 1.57529835885969992e-02
            -2.05938975761480042e-01 3.55042554837130264e-01 1.52859058974919998e-02
            -4.10759240703882977e-01 5.03034989348207461e-17 1.51760987099460003e-02
            -4.07413459425605695e-01 2.31952747352193567e-01 1.45391577748200003e-02
            -3.39061499616248097e-01 3.47894013676684599e-01 1.43739422110259994e-02
            -4.73476035944830287e-01 1.15289529048017009e-01 1.43190080035430004e-02
            -1.36112001834573365e-01 4.68200229648335187e-01 1.45429014476019993e-02
            -2.69695952756031310e-01 4.60737849124308707e-01 1.38491993647810001e-02
            -5.35876124992857972e-01 6.56258981211990512e-17 1.36990486693940004e-02
            -4.65571173408011962e-01 3.38570570114858915e-01 1.30599870310389998e-02
            -5.31141040575525469e-01 2.25574144230106549e-01 1.30613613115730005e-02
            -6.72417531977656108e-02 5.74440508138183925e-01 1.34614859364179992e-02
            -3.97956031410485578e-01 4.49424057158134993e-01 1.27215172670719998e-02
            -2.00448439189885197e-01 5.68008713815135624e-01 1.29993267975569999e-02
            -5.93093802694955041e-01 1.12268509614250958e-01 1.26842903194569999e-02
            -3.29672413780715712e-01 5.55928518197362664e-01 1.20887136126920005e-02
            -6.49777783581100055e-01 7.95748282819655808e-17 1.19178629665440006e-02
            -5.81891504668867632e-01 3.29742771775893895e-01 1.14262134394630000e-02
            -5.17334353214284337e-01 4.36775923227416152e-01 1.12022452541259993e-02
            -6.43225850131183940e-01 2.20685923529787775e-01 1.13116950123369996e-02
            -1.32108484141322285e-01 6.67226611391283475e-01 1.18542238867049999e-02
            -4.51506633325892304e-01 5.40330996613102799e-01 1.07424198339379998e-02
            -2.61596582514102172e-01 6.55784353180901225e-01 1.11824663963750008e-02
            -6.99348217842765574e-01 1.10582022959264545e-01 1.08065485960600009e-02
            -3.85413206064164238e-01 6.38487336212394729e-01 1.00777754150390003e-02
            -7.48409155164684980e-01 9.16536876325006246e-17 9.94970923213200041e-03
            -6.22546872015464325e-01 4.28708611861134903e-01 9.46846143336499933e-03
            -6.83335846938037617e-01 3.26076854336383393e-01 9.69271546586900068e-03
            -6.51137490201266211e-02 7.55917110647262303e-01 1.04304334004890007e-02
            -5.60233753993043093e-01 5.27025882085813357e-01 8.99253670308900084e-03
            -7.39491650734243500e-01 2.19794840211143289e-01 9.48739923846200067e-03
            -1.94006373119737618e-01 7.46712952846628086e-01 9.99431725244900003e-03
            -4.98507942383859537e-01 6.20290246421132396e-01 8.53996870122200082e-03
            -3.18637769847352648e-01 7.29363550790668613e-01 9.11940264582200046e-03
            -7.88670089677898312e-01 1.10644612887255495e-01 8.87521055623300033e-03
            -8.29348135045060997e-01 1.01565853896176072e-16 7.99030134096199995e-03
            -7.10305225499725079e-01 4.28481870967066858e-01 8.05074749614799988e-03
            -4.34456125396177395e-01 7.07721657126255899e-01 7.78459653634299992e-03
            -6.48737320285740937e-01 5.24615490016243080e-01 7.39990007382800019e-03
            -7.68619846181405664e-01 3.25791952380782845e-01 8.13334299876299939e-03
            -1.28090154189880373e-01 8.26262739400246415e-01 8.60000475803800006e-03
            -8.19536605836078902e-01 2.18023103014843234e-01 7.71027751666599967e-03
            -2.53414327379379967e-01 8.10569752993718362e-01 7.98250281608500062e-03
            -5.91281919080588025e-01 6.12984374028710532e-01 6.80887815498300037e-03
            -8.61399635652120743e-01 1.07538029444819910e-01 6.95469195100400020e-03
            -3.71965010291021692e-01 7.87137715992152098e-01 6.90081324771099971e-03
            -5.31113707155597803e-01 6.93708962570089738e-01 6.08886552220200012e-03
            -7.20202177923419562e-01 5.27315480709408146e-01 6.42604635464300022e-03
            -7.84396447206010317e-01 4.26844602778035598e-01 6.76111929125100021e-03
            -6.36370689277423185e-02 8.92150826573859690e-01 7.01676086174300023e-03
            -8.94465956289006048e-01 1.09540487031560762e-16 6.01388939749199995e-03
            -8.39751369848549034e-01 3.19759394235543448e-01 6.47718814993200002e-03
            -1.89682853842055216e-01 8.79917616745109044e-01 6.66166852721499989e-03
            -4.73550397830834580e-01 7.67029127592730231e-01 5.37527515541099971e-03
            -6.58111851167667128e-01 6.21569854603837180e-01 5.86825802864900017e-03
            -8.84259313777354783e-01 2.09142904866452234e-01 5.82014553803700003e-03
            -3.11279440534071306e-01 8.56760841607447921e-01 5.90210411918599964e-03
            -5.92109681606746707e-01 7.06686865853840662e-01 5.05468425581599972e-03
            -9.18643348946508898e-01 1.01372037057271214e-01 4.96098634907999984e-03
            -4.19343957925062016e-01 8.31025602200505031e-01 4.53786299893900024e-03
            -7.69384125047385292e-01 5.40958707674880412e-01 4.63278203590699982e-03
            -8.27976504829669668e-01 4.49799092322440031e-01 4.38885090972099962e-03
            -1.26274395717196425e-01 9.35289724407231127e-01 5.10983025173800034e-03
            -9.44058125619724020e-01 1.15613776174924482e-16 4.11252465550999979e-03
            -8.73944275221654299e-01 3.57990440926378273e-01 4.11305156997800007e-03
            -5.32231982057495112e-01 7.80733099630829597e-01 4.15891740887000018e-03
            -7.08354641217884606e-01 6.30786333530879206e-01 4.14019006363500024e-03
            -2.50270052934766674e-01 9.14999958338325481e-01 4.67968681326700036e-03
            -9.12753181626832388e-01 2.59510477251467364e-01 4.04996491839500032e-03
            -9.44506998707056344e-01 1.55997949457722607e-01 3.53989175112300004e-03
            -3.65963019870668205e-01 8.85544532448380606e-01 3.73576961318099979e-03
            -6.43582590393861120e-01 7.10642611351690157e-01 3.42295233433300003e-03
            -4.78978173245638184e-01 8.40645262192400322e-01 3.16317387453199979e-03
            -9.67702272601057123e-01 6.25224173268100758e-02 2.65823741461599988e-03
            -5.89697586066004864e-01 7.76609310273313702e-01 2.67617892018900006e-03
            -8.19917784356321011e-01 5.28181902224525657e-01 2.70279797596600013e-03
            -6.25582079264619118e-02 9.74488040178466686e-01 3.32356290236999986e-03
            -8.71294574610500083e-01 4.41925297246777760e-01 2.71758706882600006e-03
            -7.64875803409653998e-01 6.07977409475020658e-01 2.55934909111300022e-03
            -9.14816945191666897e-01 3.44367015105893737e-01 2.80761613499500018e-03
            -1.86663214982910303e-01 9.59766326541879766e-01 3.13674349357900015e-03
            -9.51580213576980749e-01 2.36314820624810074e-01 2.60715986157099983e-03
            -3.06052920318043875e-01 9.31974305341974674e-01 2.66902374090400001e-03
            -7.00477868969778150e-01 6.89249210565360038e-01 2.34081029261299983e-03
            -4.21337094243707355e-01 8.90449487847489007e-01 2.15399144859500019e-03
            -9.77931349120474303e-01 1.29153676091683739e-01 1.88963648898100010e-03
            -9.87862253555504055e-01 1.20978234681523903e-16 1.81726874741800003e-03
            -5.34151205565022158e-01 8.32524912275056428e-01 1.74199341019399992e-03
            -6.40293442886031183e-01 7.57551099039111331e-01 1.32907094839699994e-03
            -8.42644153128901618e-01 5.29785435489457446e-01 1.41949133903299996e-03
            -1.23927397776883871e-01 9.87850182636107688e-01 1.39942996432999996e-03
            -7.74838581919999547e-01 6.25197959050785790e-01 1.28816548380300002e-03
            -9.00562854921805989e-01 4.24677346474962802e-01 1.33942398695400004e-03
            -9.45240136607224835e-01 3.13244549623881818e-01 1.28981769243700006e-03
            -2.45183344909571554e-01 9.65393646502639591e-01 1.25297072312499994e-03
            -3.62818218907764445e-01 9.28395466735493557e-01 1.03110146672799998e-03
            -9.77250751076047086e-01 1.97502329192229098e-01 9.64757667509000012e-04
            -9.95287566748883301e-01 6.81285415252921855e-02 8.26725988317999973e-04
            -4.77242103183890709e-01 8.76088303347900776e-01 7.88848272361000052e-04
            -7.01958362179900042e-01 7.09375701626268507e-01 6.75013383016000016e-04
            -5.90879883000531292e-01 8.04958502622752703e-01 5.68376940729000009e-04
            -7.00959649866140094e-02 -1.22013963230467842e-01 1.69963608751830010e-02
            -2.09085990408722944e-01 -1.20758856025446781e-01 1.65293417033620001e-02
            -1.39092912392518225e-01 -2.41267231297798396e-01 1.63464538735140011e-02
            -2.75753624248552542e-01 -2.37572881027699662e-01 1.56536808532279983e-02
            -3.44426178957883411e-01 -1.18386788174632288e-01 1.56209624051599997e-02
            -6.90542086154240231e-02 -3.58839697664831925e-01 1.57529835885969992e-02
            -2.05938975761480125e-01 -3.55042554837130209e-01 1.52859058974919998e-02
            -4.07413459425605751e-01 -2.31952747352193456e-01 1.45391577748200003e-02
            -3.39061499616248208e-01 -3.47894013676684488e-01 1.43739422110259994e-02
            -4.73476035944830342e-01 -1.15289529048016884e-01 1.43190080035430004e-02
            -1.36112001834573698e-01 -4.68200229648335076e-01 1.45429014476019993e-02
            -2.69695952756031365e-01 -4.60737849124308652e-01 1.38491993647810001e-02
            -4.65571173408012018e-01 -3.38570570114858749e-01 1.30599870310389998e-02
            -5.31141040575525580e-01 -2.25574144230106438e-01 1.30613613115730005e-02
            -6.72417531977654997e-02 -5.74440508138183925e-01 1.34614859364179992e-02
            -3.97956031410485744e-01 -4.49424057158134882e-01 1.27215172670719998e-02
            -2.00448439189885308e-01 -5.68008713815135624e-01 1.29993267975569999e-02
            -5.93093802694955041e-01 -1.12268509614250805e-01 1.26842903194569999e-02
            -3.29672413780715823e-01 -5.55928518197362553e-01 1.20887136126920005e-02
            -5.81891504668867743e-01 -3.29742771775893784e-01 1.14262134394630000e-02
            -5.17334353214284337e-01 -4.36775923227416096e-01 1.12022452541259993e-02
            -6.43225850131184052e-01 -2.20685923529787609e-01 1.13116950123369996e-02
            -1.32108484141322757e-01 -6.67226611391283475e-01 1.18542238867049999e-02
            -4.51506633325892193e-01 -5.40330996613102910e-01 1.07424198339379998e-02
            -2.61596582514102338e-01 -6.55784353180901225e-01 1.11824663963750008e-02
            -6.99348217842765574e-01 -1.10582022959264378e-01 1.08065485960600009e-02
            -3.85413206064164404e-01 -6.38487336212394618e-01 1.00777754150390003e-02
            -6.22546872015464436e-01 -4.28708611861134736e-01 9.46846143336499933e-03
            -6.83335846938037728e-01 -3.26076854336383226e-01 9.69271546586900068e-03
            -6.51137490201264685e-02 -7.55917110647262303e-01 1.04304334004890007e-02
            -5.60233753993043204e-01 -5.27025882085813135e-01 8.99253670308900084e-03
            -7.39491650734243611e-01 -2.19794840211143122e-01 9.48739923846200067e-03
            -1.94006373119737507e-01 -7.46712952846628086e-01 9.99431725244900003e-03
            -4.98507942383859981e-01 -6.20290246421132063e-01 8.53996870122200082e-03
            -3.18637769847353147e-01 -7.29363550790668502e-01 9.11940264582200046e-03
            -7.88670089677898312e-01 -1.10644612887255286e-01 8.87521055623300033e-03
            -7.10305225499725190e-01 -4.28481870967066691e-01 8.05074749614799988e-03
            -4.34456125396177284e-01 -7.07721657126255899e-01 7.78459653634299992e-03
            -6.48737320285741159e-01 -5.24615490016242969e-01 7.39990007382800019e-03
            -7.68619846181405775e-01 -3.25791952380782623e-01 8.13334299876299939e-03
            -1.28090154189880373e-01 -8.26262739400246415e-01 8.60000475803800006e-03
            -8.19536605836078902e-01 -2.18023103014843039e-01 7.71027751666599967e-03
            -2.53414327379380355e-01 -8.10569752993718251e-01 7.98250281608500062e-03
            -5.91281919080588136e-01 -6.12984374028710421e-01 6.80887815498300037e-03
            -8.61399635652120743e-01 -1.07538029444819702e-01 6.95469195100400020e-03
            -3.71965010291021914e-01 -7.87137715992151987e-01 6.90081324771099971e-03
            -5.31113707155598247e-01 -6.93708962570089405e-01 6.08886552220200012e-03
            -7.20202177923419784e-01 -5.27315480709407924e-01 6.42604635464300022e-03
            -7.84396447206010428e-01 -4.26844602778035376e-01 6.76111929125100021e-03
            -6.36370689277421381e-02 -8.92150826573859690e-01 7.01676086174300023e-03
            -8.39751369848549145e-01 -3.19759394235543282e-01 6.47718814993200002e-03
            -1.89682853842055632e-01 -8.79917616745108933e-01 6.66166852721499989e-03
            -4.73550397830834802e-01 -7.67029127592730231e-01 5.37527515541099971e-03
            -6.58111851167667350e-01 -6.21569854603836958e-01 5.86825802864900017e-03
            -8.84259313777354894e-01 -2.09142904866452012e-01 5.82014553803700003e-03
            -3.11279440534071528e-01 -8.56760841607447921e-01 5.90210411918599964e-03
            -5.92109681606746929e-01 -7.06686865853840551e-01 5.05468425581599972e-03
            -9.18643348946508898e-01 -1.01372037057270992e-01 4.96098634907999984e-03
            -4.19343957925062183e-01 -8.31025602200504809e-01 4.53786299893900024e-03
            -7.69384125047385403e-01 -5.40958707674880190e-01 4.63278203590699982e-03
            -8.27976504829669779e-01 -4.49799092322439809e-01 4.38885090972099962e-03
            -1.26274395717196841e-01 -9.35289724407231016e-01 5.10983025173800034e-03
            -8.73944275221654410e-01 -3.57990440926378051e-01 4.11305156997800007e-03
            -5.32231982057495223e-01 -7.80733099630829486e-01 4.15891740887000018e-03
            -7.08354641217884828e-01 -6.30786333530879095e-01 4.14019006363500024e-03
            -2.50270052934766507e-01 -9.14999958338325592e-01 4.67968681326700036e-03
            -9.12753181626832388e-01 -2.59510477251467087e-01 4.04996491839500032e-03
            -9.44506998707056344e-01 -1.55997949457722357e-01 3.53989175112300004e-03
            -3.65963019870668205e-01 -8.85544532448380606e-01 3.73576961318099979e-03
            -6.43582590393861342e-01 -7.10642611351689935e-01 3.42295233433300003e-03
            -4.78978173245638350e-01 -8.40645262192400211e-01 3.16317387453199979e-03
            -9.67702272601057123e-01 -6.25224173268098399e-02 2.65823741461599988e-03
            -5.89697586066005419e-01 -7.76609310273313258e-01 2.67617892018900006e-03
            -8.19917784356321233e-01 -5.28181902224525435e-01 2.70279797596600013e-03
            -6.25582079264619395e-02 -9.74488040178466686e-01 3.32356290236999986e-03
            -8.71294574610500194e-01 -4.41925297246777593e-01 2.71758706882600006e-03
            -7.64875803409654109e-01 -6.07977409475020547e-01 2.55934909111300022e-03
            -9.14816945191667008e-01 -3.44367015105893515e-01 2.80761613499500018e-03
            -1.86663214982910303e-01 -9.59766326541879766e-01 3.13674349357900015e-03
            -9.51580213576980860e-01 -2.36314820624809824e-01 2.60715986157099983e-03
            -3.06052920318043653e-01 -9.31974305341974674e-01 2.66902374090400001e-03
            -7.00477868969778261e-01 -6.89249210565359816e-01 2.34081029261299983e-03
            -4.21337094243707966e-01 -8.90449487847488785e-01 2.15399144859500019e-03
            -9.77931349120474414e-01 -1.29153676091683517e-01 1.88963648898100010e-03
            -5.34151205565022269e-01 -8.32524912275056317e-01 1.74199341019399992e-03
            -6.40293442886031738e-01 -7.57551099039110887e-01 1.32907094839699994e-03
            -8.42644153128901729e-01 -5.29785435489457335e-01 1.41949133903299996e-03
            -1.23927397776884329e-01 -9.87850182636107577e-01 1.39942996432999996e-03
            -7.74838581919999769e-01 -6.25197959050785568e-01 1.28816548380300002e-03
            -9.00562854921806100e-01 -4.24677346474962580e-01 1.33942398695400004e-03
            -9.45240136607224946e-01 -3.13244549623881541e-01 1.28981769243700006e-03
            -2.45183344909571360e-01 -9.65393646502639702e-01 1.25297072312499994e-03
            -3.62818218907764445e-01 -9.28395466735493557e-01 1.03110146672799998e-03
            -9.77250751076047086e-01 -1.97502329192228876e-01 9.64757667509000012e-04
            -9.95287566748883301e-01 -6.81285415252919496e-02 8.26725988317999973e-04
            -4.77242103183890931e-01 -8.76088303347900665e-01 7.88848272361000052e-04
            -7.01958362179900264e-01 -7.09375701626268285e-01 6.75013383016000016e-04
            -5.90879883000531847e-01 -8.04958502622752259e-01 5.68376940729000009e-04
            ];

    case 53

        subcase=1;

        switch subcase
            case 1

                xyw=[
                    0.00000000000000000e+00 -0.00000000000000000e+00 1.48262171762400009e-02
                    1.30474371960067004e-01 0.00000000000000000e+00 1.46596687024809994e-02
                    1.94648332574785976e-01 1.12291657506824530e-01 1.43293711259989999e-02
                    2.58739465066547158e-01 -1.40828204556622008e-04 1.41657314044679999e-02
                    3.20738724375208728e-01 1.10839917686088585e-01 1.36793777585130002e-02
                    2.56760117522281506e-01 2.21981783654961629e-01 1.36800980112599998e-02
                    3.82589248365508172e-01 -4.21249954902817088e-04 1.33610901887220006e-02
                    3.79556477645500878e-01 2.18756869463238662e-01 1.28865763702140004e-02
                    4.41221678194768319e-01 1.08803363982429485e-01 1.27309586541619994e-02
                    3.15693493366070055e-01 3.27274024769794170e-01 1.27335707412020004e-02
                    4.99815383491732101e-01 -6.16701742984842043e-04 1.22728977091429992e-02
                    4.95462823336197511e-01 2.14797146808032663e-01 1.18104934192140004e-02
                    4.34031753344188032e-01 3.22020991294299841e-01 1.18119862010690000e-02
                    5.53746905087918795e-01 1.06715830316165577e-01 1.15169048960539998e-02
                    3.70117392358726816e-01 4.26539150975871806e-01 1.15199313878869992e-02
                    6.08191643411647553e-01 -1.18608949123721050e-04 1.09395418124159996e-02
                    5.43898298600371422e-01 3.16575196494049804e-01 1.06252907191070004e-02
                    6.01780513525812522e-01 2.11428840159785136e-01 1.04907059906140000e-02
                    4.82583818245495566e-01 4.19517364611189658e-01 1.04828891908719992e-02
                    6.55725371250924116e-01 1.05974454033243348e-01 1.00796115836390004e-02
                    4.18220534967930047e-01 5.18594064444639069e-01 1.00794828416590002e-02
                    7.05406489020722671e-01 2.78898551412186217e-03 9.40845773050300038e-03
                    6.41847411671825907e-01 3.13928034720394100e-01 9.24078299032900057e-03
                    5.84112586053404881e-01 4.14406197086055028e-01 9.20593261322900057e-03
                    6.95296263473091369e-01 2.11575374477000033e-01 8.98950156821000032e-03
                    5.22602457173299895e-01 5.11351508517829845e-01 8.93661852393799923e-03
                    7.44215180263244624e-01 1.09725372226434192e-01 8.46634636712599933e-03
                    4.56616708998531295e-01 6.03463604140960230e-01 8.46188329973500061e-03
                    6.71323038532189886e-01 4.14477478633195007e-01 7.88432596683500027e-03
                    7.89047286015875593e-01 1.25120318935048810e-02 7.72106443703700004e-03
                    7.25479022634508808e-01 3.16099367730773795e-01 7.80084606203599977e-03
                    6.12562816884762884e-01 5.09461921174128807e-01 7.67590729036300018e-03
                    7.74065301586715493e-01 2.16780409637637034e-01 7.37352560194999996e-03
                    5.49127368076386402e-01 5.99361015087210069e-01 7.28199558876000026e-03
                    8.17488908432171368e-01 1.20530581019056032e-01 6.67061502106700011e-03
                    4.77852139278117150e-01 6.83471965190333153e-01 6.75288352783299985e-03
                    7.54416599766813079e-01 3.99635061416432846e-01 6.52417810093700035e-03
                    6.98936076916079729e-01 4.94098393918156409e-01 6.47039073583199984e-03
                    8.03607036320060586e-01 3.02942881833024702e-01 6.23376213407899997e-03
                    8.58149677591916626e-01 3.64012824514318156e-02 6.02765717185200015e-03
                    6.38000674534805690e-01 5.84104846145397150e-01 6.19955603562300037e-03
                    8.45088040403722873e-01 2.08415843454622718e-01 5.38752222495999990e-03
                    5.68982746808920692e-01 6.68636542890197361e-01 5.80751036954100033e-03
                    8.82205700579757446e-01 1.32529256315106231e-01 4.53486784658799994e-03
                    4.89780411931682369e-01 7.46853144026378590e-01 5.26685028270500004e-03
                    7.74824182574610054e-01 4.72949181840120580e-01 5.16661701956700001e-03
                    8.26255760763167313e-01 3.78430440686862712e-01 5.15119225902399992e-03
                    7.17187801767041999e-01 5.63067733197873732e-01 5.01024438852399993e-03
                    9.11502810785417061e-01 5.43586922148707116e-02 4.13473702879899995e-03
                    8.70057704500828644e-01 2.80078934710390526e-01 4.81739668099000033e-03
                    6.51414015950670611e-01 6.48286404060328780e-01 4.74850298747600021e-03
                    5.74983693670289231e-01 7.27883554690509582e-01 4.40448832080200022e-03
                    9.10685618687926524e-01 1.90862858470398244e-01 4.18053394324000016e-03
                    4.90211076307295435e-01 7.99855578193879646e-01 3.78032759303999991e-03
                    9.42574225903891061e-01 9.98504687650996758e-02 3.55846386921100004e-03
                    8.49945947579159866e-01 4.24378934105072991e-01 3.85502295556000010e-03
                    7.96442587553838677e-01 5.18396675002939067e-01 3.90916037461699981e-03
                    8.94485706406168091e-01 3.26232138860153731e-01 3.69682193337999990e-03
                    7.33377823982738875e-01 6.09237805586694825e-01 3.79249084190899990e-03
                    6.59047676212604627e-01 6.95009289784345508e-01 3.54570802098299991e-03
                    9.34147321142855369e-01 2.32217090667899445e-01 3.09316883568099997e-03
                    5.74110913371624898e-01 7.73039020653807807e-01 3.17774833646300003e-03
                    9.67637049737522514e-01 1.03473668545183122e-02 2.87756055216200008e-03
                    9.64502093810197580e-01 1.36355425400169178e-01 2.49034745475699981e-03
                    8.61476379068770948e-01 4.66013348233383751e-01 2.60136723227800005e-03
                    9.09269388822445568e-01 3.65423092540127725e-01 2.48857966320900018e-03
                    8.02497923796285040e-01 5.62525890866345968e-01 2.59241207829399986e-03
                    7.31706836312515208e-01 6.54343366498676637e-01 2.44547418725300000e-03
                    6.49566298031229006e-01 7.38653020286580309e-01 2.22077428728399999e-03
                    9.47447264714400328e-01 2.67392496132636937e-01 1.89187832491400005e-03
                    5.55200958450328241e-01 8.14155109442827540e-01 2.00791878619099992e-03
                    9.86324819696963395e-01 4.89925509034932899e-02 1.74919241129499990e-03
                    9.75540084180159361e-01 1.73756058546072317e-01 1.38941196409599998e-03
                    9.07542210175635300e-01 4.10537714094330597e-01 1.13248117881799997e-03
                    8.54722554207692276e-01 5.11570605698023151e-01 1.14561860890600000e-03
                    7.89736277485199700e-01 6.07381765841921606e-01 1.10982691108799992e-03
                    7.12932729686729605e-01 6.96484098600620327e-01 1.01516445954099991e-03
                    9.48074672904392401e-01 3.07591367452174080e-01 9.18301477314999973e-04
                    6.24613680068103472e-01 7.77110308150813722e-01 9.19438803836000003e-04
                    5.26887418233451643e-01 8.46818978589414351e-01 8.14130778808999970e-04
                    9.94041065454129624e-01 8.65416725291536792e-02 6.74237438331999985e-04
                    9.77248171831717083e-01 2.05729323441692480e-01 4.41837214846999991e-04
                    6.52371859800335158e-02 1.12994120660238062e-01 1.46596687024809994e-02
                    7.67382534213987302e-05 2.24716229567458986e-01 1.43293711259989999e-02
                    1.29491693335989033e-01 2.24004535606947830e-01 1.41657314044679999e-02
                    6.43791777180756225e-02 3.33187842129390177e-01 1.36793777585130002e-02
                    -6.38618050614372518e-02 3.33351676280454590e-01 1.36800980112599998e-02
                    1.91659437345043038e-01 3.31121383321872631e-01 1.33610901887220006e-02
                    3.29232615229496845e-04 4.38083986543563508e-01 1.28865763702140004e-02
                    1.26384361871395429e-01 4.36510864008286592e-01 1.27309586541619994e-02
                    -1.25580872766384299e-01 4.37035597449367930e-01 1.27335707412020004e-02
                    2.50441771121849122e-01 4.32544468434608809e-01 1.22728977091429992e-02
                    6.17116258719269534e-02 5.36481965043924758e-01 1.18104934192140004e-02
                    -6.18624823406171143e-02 5.36893020092318207e-01 1.18119862010690000e-02
                    1.84454832504210547e-01 5.32916802231230835e-01 1.15169048960539998e-02
                    -1.84335044274387544e-01 5.33800639653045805e-01 1.15199313878869992e-02
                    3.04198540068881196e-01 5.26650109089331520e-01 1.09395418124159996e-02
                    -2.21301307171161516e-03 6.29317341910080708e-01 1.06252907191070004e-02
                    1.17787510091852884e-01 6.26871632295691206e-01 1.04907059906140000e-02
                    -1.22020785959241249e-01 6.27688528361486253e-01 1.04828891908719992e-02
                    2.36086116280487118e-01 6.20862056425904085e-01 1.00796115836390004e-02
                    -2.40005366576916651e-01 6.21486639888864989e-01 1.00794828416590002e-02
                    3.50287912204345020e-01 6.12294432243395415e-01 9.40845773050300038e-03
                    4.90540528079284541e-02 7.12820181221286875e-01 9.24078299032900057e-03
                    -6.68300011355219270e-02 7.13059436735500096e-01 9.20593261322900057e-03
                    1.64418482624257872e-01 7.07931914562595299e-01 8.98950156821000032e-03
                    -1.81542168053285435e-01 7.08262758251161695e-01 8.93661852393799923e-03
                    2.77082630343826963e-01 6.99371938103202306e-01 8.46634636712599933e-03
                    -2.94306456946122008e-01 6.97173471855654703e-01 8.46188329973500061e-03
                    -2.32865065267737759e-02 7.88621544831233479e-01 7.88432596683500027e-03
                    3.83687905535201546e-01 6.89591010423666551e-01 7.72106443703700004e-03
                    8.89894287422054453e-02 7.86332947379577263e-01 7.80084606203599977e-03
                    -1.34925557555239245e-01 7.85225921423024387e-01 7.67590729036300018e-03
                    1.99295309004367233e-01 7.78750420180977065e-01 7.37352560194999996e-03
                    -2.44498181065358838e-01 7.75238758211043710e-01 7.28199558876000026e-03
                    3.04361909120684893e-01 7.68231452523799097e-01 6.67061502106700011e-03
                    -3.52978014990243394e-01 7.55568074462755868e-01 6.75288352783299985e-03
                    3.11141844538213377e-02 8.53161471142953931e-01 6.52417810093700035e-03
                    -7.84337226441739754e-02 8.52345595189837657e-01 6.47039073583199984e-03
                    1.39447286596963677e-01 8.47415549029608917e-01 6.23376213407899997e-03
                    3.97550403462685664e-01 7.61380062269741309e-01 6.02765717185200015e-03
                    -1.86849297968112005e-01 8.44577214851447988e-01 6.19955603562300037e-03
                    2.42050605218997406e-01 8.36075633151345365e-01 5.38752222495999990e-03
                    -2.94564858637053950e-01 8.27071784496673179e-01 5.80751036954100033e-03
                    3.26329147576337519e-01 8.30277176223071067e-01 4.53486784658799994e-03
                    -4.01903589657280624e-01 7.97588851022033363e-01 5.26685028270500004e-03
                    -2.21739148853051793e-02 9.07492016496184717e-01 5.16661701956700001e-03
                    8.53975051814203773e-02 9.04773699187571778e-01 5.15119225902399992e-03
                    -1.29037060117156183e-01 9.02636722213513321e-01 5.01024438852399993e-03
                    4.08675397018131159e-01 8.16563935868526936e-01 4.13473702879899995e-03
                    1.92473379726332994e-01 8.93531542211287166e-01 4.81739668099000033e-03
                    -2.35725486868972739e-01 8.88284288224686636e-01 4.74850298747600021e-03
                    -3.42873802523756377e-01 8.61892262825535105e-01 4.40448832080200022e-03
                    2.90050725269684451e-01 8.84108309880092036e-01 4.18053394324000016e-03
                    -4.47589711920942324e-01 8.24463034395569672e-01 3.78032759303999991e-03
                    3.84814070421584775e-01 8.66218458967771721e-01 3.55846386921100004e-03
                    5.74500360236246121e-02 9.48264249499725764e-01 3.85502295556000010e-03
                    -5.07233960130112557e-02 9.48937850978905817e-01 3.90916037461699981e-03
                    1.64717533419258572e-01 9.37763414499887427e-01 3.69682193337999990e-03
                    -1.60926504592593195e-01 9.39742728934551796e-01 3.79249084190899990e-03
                    -2.72371862713121304e-01 9.18256674797389705e-01 3.54570802098299991e-03
                    2.65967760860112534e-01 9.25103856320842688e-01 3.09316883568099997e-03
                    -3.82415973317028368e-01 8.83714145896618231e-01 3.17774833646300003e-03
                    4.74857442310471445e-01 8.43171950142979965e-01 2.87756055216200008e-03
                    3.64163784564718485e-01 9.03461027942997319e-01 2.49034745475699981e-03
                    2.71587914616313701e-02 9.79067103150480289e-01 2.60136723227800005e-03
                    1.38169013142000363e-01 9.70161935873852022e-01 2.48857966320900018e-03
                    -8.59127498785856036e-02 9.76246533925024473e-01 2.59241207829399986e-03
                    -2.00824560029427612e-01 9.60848391618718378e-01 2.44547418725300000e-03
                    -3.14909131134666176e-01 9.31867425680548367e-01 2.22077428728399999e-03
                    2.42154937925004393e-01 9.54209648055068915e-01 1.89187832491400005e-03
                    -4.27478528173224182e-01 8.87895688944866701e-01 2.00791878619099992e-03
                    4.50733616169854467e-01 8.78678625692422943e-01 1.74919241129499990e-03
                    3.37292881327725003e-01 9.31720524583063869e-01 1.38941196409599998e-03
                    9.82350154705347034e-02 9.91223466065941694e-01 1.13248117881799997e-03
                    -1.56718632600340441e-02 9.95996747980394947e-01 1.14561860890600000e-03
                    -1.31139900271955601e-01 9.87622561513300301e-01 1.10982691108799992e-03
                    -2.46706557876678140e-01 9.65659904398402213e-01 1.01516445954099991e-03
                    2.07655398253819568e-01 9.74852435145913021e-01 9.18301477314999973e-04
                    -3.60690428367306137e-01 9.29486468565670299e-01 9.19438803836000003e-04
                    -4.69923038748497612e-01 8.79707378419272601e-01 8.14130778808999970e-04
                    4.22073245830823951e-01 9.04135651352802938e-01 6.74237438331999985e-04
                    3.10457265511967562e-01 9.49186404329013578e-01 4.41837214846999991e-04
                    -6.52371859800334741e-02 1.12994120660238076e-01 1.46596687024809994e-02
                    -1.94571594321364560e-01 1.12424572060634498e-01 1.43293711259989999e-02
                    -1.29247771730558125e-01 2.24145363811504544e-01 1.41657314044679999e-02
                    -2.56359546657133119e-01 2.22347924443301648e-01 1.36793777585130002e-02
                    -3.20621922583718744e-01 1.11369892625493003e-01 1.36800980112599998e-02
                    -1.90929811020465190e-01 3.31542633276775522e-01 1.33610901887220006e-02
                    -3.79227245030271387e-01 2.19327117080325013e-01 1.28865763702140004e-02
                    -3.14837316323372862e-01 3.27707500025857190e-01 1.27309586541619994e-02
                    -4.41274366132454410e-01 1.09761572679573816e-01 1.27335707412020004e-02
                    -2.49373612369882869e-01 4.33161170177593813e-01 1.22728977091429992e-02
                    -4.33751197464270544e-01 3.21684818235892178e-01 1.18104934192140004e-02
                    -4.95894235684805174e-01 2.14872028798018533e-01 1.18119862010690000e-02
                    -3.69292072583708275e-01 4.26200971915065341e-01 1.15169048960539998e-02
                    -5.54452436633114498e-01 1.07261488677174138e-01 1.15199313878869992e-02
                    -3.03993103342766302e-01 5.26768718038455375e-01 1.09395418124159996e-02
                    -5.46111311672083155e-01 3.12742145416031014e-01 1.06252907191070004e-02
                    -4.83993003433959790e-01 4.15442792135906125e-01 1.04907059906140000e-02
                    -6.04604604204736940e-01 2.08171163750296623e-01 1.04828891908719992e-02
                    -4.19639254970436970e-01 5.14887602392660848e-01 1.00796115836390004e-02
                    -6.58225901544846614e-01 1.02892575444226295e-01 1.00794828416590002e-02
                    -3.55118576816377540e-01 6.09505446729273648e-01 9.40845773050300038e-03
                    -5.92793358863897279e-01 3.98892146500893052e-01 9.24078299032900057e-03
                    -6.50942587188926836e-01 2.98653239649445290e-01 9.20593261322900057e-03
                    -5.30877780848833303e-01 4.96356540085595488e-01 8.98950156821000032e-03
                    -7.04144625226585386e-01 1.96911249733332044e-01 8.93661852393799923e-03
                    -4.67132549919417717e-01 5.89646565876768114e-01 8.46634636712599933e-03
                    -7.50923165944653359e-01 9.37098677146946529e-02 8.46188329973500061e-03
                    -6.94609545058963551e-01 3.74144066198038749e-01 7.88432596683500027e-03
                    -4.05359380480673881e-01 6.77078978530161857e-01 7.72106443703700004e-03
                    -6.36489593892303529e-01 4.70233579648803579e-01 7.80084606203599977e-03
                    -7.47488374440002268e-01 2.75764000248895524e-01 7.67590729036300018e-03
                    -5.74769992582348177e-01 5.61970010543340170e-01 7.37352560194999996e-03
                    -7.93625549141745323e-01 1.75877743123833558e-01 7.28199558876000026e-03
                    -5.13126999311486531e-01 6.47700871504743314e-01 6.67061502106700011e-03
                    -8.30830154268360599e-01 7.20961092724230201e-02 6.75288352783299985e-03
                    -7.23302415312991553e-01 4.53526409726521307e-01 6.52417810093700035e-03
                    -7.77369799560253760e-01 3.58247201271681304e-01 6.47039073583199984e-03
                    -6.64159749723096771e-01 5.44472667196584492e-01 6.23376213407899997e-03
                    -4.60599274129230796e-01 7.24978779818309604e-01 6.02765717185200015e-03
                    -8.24849972502917805e-01 2.60472368706050783e-01 6.19955603562300037e-03
                    -6.03037435184725190e-01 6.27659789696722981e-01 5.38752222495999990e-03
                    -8.63547605445974642e-01 1.58435241606475985e-01 5.80751036954100033e-03
                    -5.55876553003419871e-01 6.97747919907965031e-01 4.53486784658799994e-03
                    -8.91684001588963215e-01 5.07357069956547516e-02 5.26685028270500004e-03
                    -7.96998097459915233e-01 4.34542834656064192e-01 5.16661701956700001e-03
                    -7.40858255581746783e-01 5.26343258500709399e-01 5.15119225902399992e-03
                    -8.46224861884198098e-01 3.39568989015639922e-01 5.01024438852399993e-03
                    -5.02827413767285569e-01 7.62205243653656384e-01 4.13473702879899995e-03
                    -6.77584324774495594e-01 6.13452607500896807e-01 4.81739668099000033e-03
                    -8.87139502819643266e-01 2.39997884164358133e-01 4.74850298747600021e-03
                    -9.17857496194045663e-01 1.34008708135025634e-01 4.40448832080200022e-03
                    -6.20634893418241962e-01 6.93245451409693847e-01 4.18053394324000016e-03
                    -9.37800788228237980e-01 2.46074562016901158e-02 3.78032759303999991e-03
                    -5.57760155482306175e-01 7.66367990202672322e-01 3.55846386921100004e-03
                    -7.92495911555535337e-01 5.23885315394652995e-01 3.85502295556000010e-03
                    -8.47165983566850023e-01 4.30541175975966750e-01 3.90916037461699981e-03
                    -7.29768172986909658e-01 6.11531275639733862e-01 3.69682193337999990e-03
                    -8.94304328575332153e-01 3.30504923347856971e-01 3.79249084190899990e-03
                    -9.31419538925726043e-01 2.23247385013044614e-01 3.54570802098299991e-03
                    -6.68179560282742613e-01 6.92886765652943604e-01 3.09316883568099997e-03
                    -9.56526886688653266e-01 1.10675125242810840e-01 3.17774833646300003e-03
                    -4.92779607427051181e-01 8.32824583288461651e-01 2.87756055216200008e-03
                    -6.00338309245479040e-01 7.67105602542828335e-01 2.49034745475699981e-03
                    -8.34317587607139810e-01 5.13053754917096705e-01 2.60136723227800005e-03
                    -7.71100375680444983e-01 6.04738843333724629e-01 2.48857966320900018e-03
                    -8.88410673674870699e-01 4.13720643058678728e-01 2.59241207829399986e-03
                    -9.32531396341942931e-01 3.06505025120041796e-01 2.44547418725300000e-03
                    -9.64475429165895348e-01 1.93214405393968031e-01 2.22077428728399999e-03
                    -7.05292326789395907e-01 6.86817151922432201e-01 1.89187832491400005e-03
                    -9.82679486623552645e-01 7.37405795020392718e-02 2.00791878619099992e-03
                    -5.35591203527109094e-01 8.29686074788929750e-01 1.74919241129499990e-03
                    -6.38247202852434414e-01 7.57964466036991635e-01 1.38941196409599998e-03
                    -8.09307194705100486e-01 5.80685751971611319e-01 1.13248117881799997e-03
                    -8.70394417467726389e-01 4.84426142282372130e-01 1.14561860890600000e-03
                    -9.20876177757155134e-01 3.80240795671379139e-01 1.10982691108799992e-03
                    -9.59639287563407772e-01 2.69175805797781831e-01 1.01516445954099991e-03
                    -7.40419274650572556e-01 6.67261067693739274e-01 9.18301477314999973e-04
                    -9.85304108435409720e-01 1.52376160414857048e-01 9.19438803836000003e-04
                    -9.96810456981949256e-01 3.28883998298581670e-02 8.14130778808999970e-04
                    -5.71967819623305673e-01 8.17593978823649370e-01 6.74237438331999985e-04
                    -6.66790906319749466e-01 7.43457080887321209e-01 4.41837214846999991e-04
                    -1.30474371960067004e-01 1.59785021991657226e-17 1.46596687024809994e-02
                    -1.94648332574785976e-01 -1.12291657506824502e-01 1.43293711259989999e-02
                    -2.58739465066547158e-01 1.40828204556673914e-04 1.41657314044679999e-02
                    -3.20738724375208728e-01 -1.10839917686088585e-01 1.36793777585130002e-02
                    -2.56760117522281506e-01 -2.21981783654961601e-01 1.36800980112599998e-02
                    -3.82589248365508172e-01 4.21249954902794266e-04 1.33610901887220006e-02
                    -3.79556477645500878e-01 -2.18756869463238718e-01 1.28865763702140004e-02
                    -4.41221678194768319e-01 -1.08803363982429471e-01 1.27309586541619994e-02
                    -3.15693493366070110e-01 -3.27274024769794170e-01 1.27335707412020004e-02
                    -4.99815383491732101e-01 6.16701742984933333e-04 1.22728977091429992e-02
                    -4.95462823336197511e-01 -2.14797146808032635e-01 1.18104934192140004e-02
                    -4.34031753344188087e-01 -3.22020991294299730e-01 1.18119862010690000e-02
                    -5.53746905087918795e-01 -1.06715830316165605e-01 1.15169048960539998e-02
                    -3.70117392358726871e-01 -4.26539150975871806e-01 1.15199313878869992e-02
                    -6.08191643411647553e-01 1.18608949123796023e-04 1.09395418124159996e-02
                    -5.43898298600371422e-01 -3.16575196494049749e-01 1.06252907191070004e-02
                    -6.01780513525812633e-01 -2.11428840159784970e-01 1.04907059906140000e-02
                    -4.82583818245495677e-01 -4.19517364611189547e-01 1.04828891908719992e-02
                    -6.55725371250924116e-01 -1.05974454033243265e-01 1.00796115836390004e-02
                    -4.18220534967930158e-01 -5.18594064444638847e-01 1.00794828416590002e-02
                    -7.05406489020722671e-01 -2.78898551412180883e-03 9.40845773050300038e-03
                    -6.41847411671825796e-01 -3.13928034720394156e-01 9.24078299032900057e-03
                    -5.84112586053404992e-01 -4.14406197086054973e-01 9.20593261322900057e-03
                    -6.95296263473091369e-01 -2.11575374476999867e-01 8.98950156821000032e-03
                    -5.22602457173299895e-01 -5.11351508517829734e-01 8.93661852393799923e-03
                    -7.44215180263244624e-01 -1.09725372226434192e-01 8.46634636712599933e-03
                    -4.56616708998531351e-01 -6.03463604140960230e-01 8.46188329973500061e-03
                    -6.71323038532189997e-01 -4.14477478633194840e-01 7.88432596683500027e-03
                    -7.89047286015875704e-01 -1.25120318935046295e-02 7.72106443703700004e-03
                    -7.25479022634508808e-01 -3.16099367730773795e-01 7.80084606203599977e-03
                    -6.12562816884763106e-01 -5.09461921174128696e-01 7.67590729036300018e-03
                    -7.74065301586715493e-01 -2.16780409637636895e-01 7.37352560194999996e-03
                    -5.49127368076386624e-01 -5.99361015087209847e-01 7.28199558876000026e-03
                    -8.17488908432171368e-01 -1.20530581019056032e-01 6.67061502106700011e-03
                    -4.77852139278117038e-01 -6.83471965190333153e-01 6.75288352783299985e-03
                    -7.54416599766813190e-01 -3.99635061416432680e-01 6.52417810093700035e-03
                    -6.98936076916079729e-01 -4.94098393918156409e-01 6.47039073583199984e-03
                    -8.03607036320060586e-01 -3.02942881833024813e-01 6.23376213407899997e-03
                    -8.58149677591916626e-01 -3.64012824514317671e-02 6.02765717185200015e-03
                    -6.38000674534805690e-01 -5.84104846145397150e-01 6.19955603562300037e-03
                    -8.45088040403722873e-01 -2.08415843454622496e-01 5.38752222495999990e-03
                    -5.68982746808920914e-01 -6.68636542890197139e-01 5.80751036954100033e-03
                    -8.82205700579757446e-01 -1.32529256315106203e-01 4.53486784658799994e-03
                    -4.89780411931682369e-01 -7.46853144026378590e-01 5.26685028270500004e-03
                    -7.74824182574610165e-01 -4.72949181840120469e-01 5.16661701956700001e-03
                    -8.26255760763167424e-01 -3.78430440686862435e-01 5.15119225902399992e-03
                    -7.17187801767042110e-01 -5.63067733197873621e-01 5.01024438852399993e-03
                    -9.11502810785417061e-01 -5.43586922148704341e-02 4.13473702879899995e-03
                    -8.70057704500828644e-01 -2.80078934710390470e-01 4.81739668099000033e-03
                    -6.51414015950670722e-01 -6.48286404060328669e-01 4.74850298747600021e-03
                    -5.74983693670289342e-01 -7.27883554690509471e-01 4.40448832080200022e-03
                    -9.10685618687926524e-01 -1.90862858470398217e-01 4.18053394324000016e-03
                    -4.90211076307295435e-01 -7.99855578193879535e-01 3.78032759303999991e-03
                    -9.42574225903891061e-01 -9.98504687650994538e-02 3.55846386921100004e-03
                    -8.49945947579159977e-01 -4.24378934105072825e-01 3.85502295556000010e-03
                    -7.96442587553838788e-01 -5.18396675002938845e-01 3.90916037461699981e-03
                    -8.94485706406168091e-01 -3.26232138860153786e-01 3.69682193337999990e-03
                    -7.33377823982738986e-01 -6.09237805586694492e-01 3.79249084190899990e-03
                    -6.59047676212604516e-01 -6.95009289784345619e-01 3.54570802098299991e-03
                    -9.34147321142855480e-01 -2.32217090667899168e-01 3.09316883568099997e-03
                    -5.74110913371624787e-01 -7.73039020653807807e-01 3.17774833646300003e-03
                    -9.67637049737522514e-01 -1.03473668545183868e-02 2.87756055216200008e-03
                    -9.64502093810197580e-01 -1.36355425400169206e-01 2.49034745475699981e-03
                    -8.61476379068770948e-01 -4.66013348233383695e-01 2.60136723227800005e-03
                    -9.09269388822445679e-01 -3.65423092540127503e-01 2.48857966320900018e-03
                    -8.02497923796285151e-01 -5.62525890866345857e-01 2.59241207829399986e-03
                    -7.31706836312515319e-01 -6.54343366498676304e-01 2.44547418725300000e-03
                    -6.49566298031229339e-01 -7.38653020286580086e-01 2.22077428728399999e-03
                    -9.47447264714400439e-01 -2.67392496132636825e-01 1.89187832491400005e-03
                    -5.55200958450328241e-01 -8.14155109442827540e-01 2.00791878619099992e-03
                    -9.86324819696963395e-01 -4.89925509034932899e-02 1.74919241129499990e-03
                    -9.75540084180159361e-01 -1.73756058546072317e-01 1.38941196409599998e-03
                    -9.07542210175635300e-01 -4.10537714094330430e-01 1.13248117881799997e-03
                    -8.54722554207692387e-01 -5.11570605698023040e-01 1.14561860890600000e-03
                    -7.89736277485199589e-01 -6.07381765841921717e-01 1.10982691108799992e-03
                    -7.12932729686729827e-01 -6.96484098600620105e-01 1.01516445954099991e-03
                    -9.48074672904392513e-01 -3.07591367452173803e-01 9.18301477314999973e-04
                    -6.24613680068103361e-01 -7.77110308150813722e-01 9.19438803836000003e-04
                    -5.26887418233451865e-01 -8.46818978589414129e-01 8.14130778808999970e-04
                    -9.94041065454129624e-01 -8.65416725291537070e-02 6.74237438331999985e-04
                    -9.77248171831717194e-01 -2.05729323441692341e-01 4.41837214846999991e-04
                    -6.52371859800335574e-02 -1.12994120660238048e-01 1.46596687024809994e-02
                    -7.67382534214761423e-05 -2.24716229567458986e-01 1.43293711259989999e-02
                    -1.29491693335989089e-01 -2.24004535606947774e-01 1.41657314044679999e-02
                    -6.43791777180756780e-02 -3.33187842129390177e-01 1.36793777585130002e-02
                    6.38618050614372934e-02 -3.33351676280454534e-01 1.36800980112599998e-02
                    -1.91659437345043204e-01 -3.31121383321872576e-01 1.33610901887220006e-02
                    -3.29232615229550513e-04 -4.38083986543563508e-01 1.28865763702140004e-02
                    -1.26384361871395567e-01 -4.36510864008286592e-01 1.27309586541619994e-02
                    1.25580872766384161e-01 -4.37035597449367930e-01 1.27335707412020004e-02
                    -2.50441771121849066e-01 -4.32544468434608864e-01 1.22728977091429992e-02
                    -6.17116258719273836e-02 -5.36481965043924758e-01 1.18104934192140004e-02
                    6.18624823406169269e-02 -5.36893020092318207e-01 1.18119862010690000e-02
                    -1.84454832504210853e-01 -5.32916802231230724e-01 1.15169048960539998e-02
                    1.84335044274387377e-01 -5.33800639653045916e-01 1.15199313878869992e-02
                    -3.04198540068881307e-01 -5.26650109089331409e-01 1.09395418124159996e-02
                    2.21301307171125825e-03 -6.29317341910080708e-01 1.06252907191070004e-02
                    -1.17787510091853245e-01 -6.26871632295691206e-01 1.04907059906140000e-02
                    1.22020785959240902e-01 -6.27688528361486364e-01 1.04828891908719992e-02
                    -2.36086116280487618e-01 -6.20862056425903863e-01 1.00796115836390004e-02
                    2.40005366576916318e-01 -6.21486639888865211e-01 1.00794828416590002e-02
                    -3.50287912204345242e-01 -6.12294432243395303e-01 9.40845773050300038e-03
                    -4.90540528079285373e-02 -7.12820181221286875e-01 9.24078299032900057e-03
                    6.68300011355213719e-02 -7.13059436735500207e-01 9.20593261322900057e-03
                    -1.64418482624258261e-01 -7.07931914562595299e-01 8.98950156821000032e-03
                    1.81542168053285491e-01 -7.08262758251161695e-01 8.93661852393799923e-03
                    -2.77082630343827352e-01 -6.99371938103202084e-01 8.46634636712599933e-03
                    2.94306456946121731e-01 -6.97173471855654814e-01 8.46188329973500061e-03
                    2.32865065267733318e-02 -7.88621544831233479e-01 7.88432596683500027e-03
                    -3.83687905535201657e-01 -6.89591010423666551e-01 7.72106443703700004e-03
                    -8.89894287422055424e-02 -7.86332947379577152e-01 7.80084606203599977e-03
                    1.34925557555238801e-01 -7.85225921423024387e-01 7.67590729036300018e-03
                    -1.99295309004367149e-01 -7.78750420180977065e-01 7.37352560194999996e-03
                    2.44498181065358422e-01 -7.75238758211043821e-01 7.28199558876000026e-03
                    -3.04361909120684948e-01 -7.68231452523799097e-01 6.67061502106700011e-03
                    3.52978014990243283e-01 -7.55568074462755868e-01 6.75288352783299985e-03
                    -3.11141844538214384e-02 -8.53161471142953931e-01 6.52417810093700035e-03
                    7.84337226441735036e-02 -8.52345595189837657e-01 6.47039073583199984e-03
                    -1.39447286596963788e-01 -8.47415549029608917e-01 6.23376213407899997e-03
                    -3.97550403462686275e-01 -7.61380062269740976e-01 6.02765717185200015e-03
                    1.86849297968111894e-01 -8.44577214851447988e-01 6.19955603562300037e-03
                    -2.42050605218997905e-01 -8.36075633151345254e-01 5.38752222495999990e-03
                    2.94564858637054061e-01 -8.27071784496673179e-01 5.80751036954100033e-03
                    -3.26329147576337797e-01 -8.30277176223070956e-01 4.53486784658799994e-03
                    4.01903589657280569e-01 -7.97588851022033363e-01 5.26685028270500004e-03
                    2.21739148853048636e-02 -9.07492016496184717e-01 5.16661701956700001e-03
                    -8.53975051814204883e-02 -9.04773699187571778e-01 5.15119225902399992e-03
                    1.29037060117155655e-01 -9.02636722213513432e-01 5.01024438852399993e-03
                    -4.08675397018131659e-01 -8.16563935868526602e-01 4.13473702879899995e-03
                    -1.92473379726333715e-01 -8.93531542211287055e-01 4.81739668099000033e-03
                    2.35725486868972628e-01 -8.88284288224686747e-01 4.74850298747600021e-03
                    3.42873802523756099e-01 -8.61892262825535216e-01 4.40448832080200022e-03
                    -2.90050725269684784e-01 -8.84108309880091925e-01 4.18053394324000016e-03
                    4.47589711920942268e-01 -8.24463034395569783e-01 3.78032759303999991e-03
                    -3.84814070421584664e-01 -8.66218458967771721e-01 3.55846386921100004e-03
                    -5.74500360236249383e-02 -9.48264249499725764e-01 3.85502295556000010e-03
                    5.07233960130107214e-02 -9.48937850978905817e-01 3.90916037461699981e-03
                    -1.64717533419258683e-01 -9.37763414499887427e-01 3.69682193337999990e-03
                    1.60926504592592667e-01 -9.39742728934551907e-01 3.79249084190899990e-03
                    2.72371862713121193e-01 -9.18256674797389705e-01 3.54570802098299991e-03
                    -2.65967760860113089e-01 -9.25103856320842577e-01 3.09316883568099997e-03
                    3.82415973317028257e-01 -8.83714145896618231e-01 3.17774833646300003e-03
                    -4.74857442310471556e-01 -8.43171950142979854e-01 2.87756055216200008e-03
                    -3.64163784564719040e-01 -9.03461027942997097e-01 2.49034745475699981e-03
                    -2.71587914616314881e-02 -9.79067103150480289e-01 2.60136723227800005e-03
                    -1.38169013142000474e-01 -9.70161935873852022e-01 2.48857966320900018e-03
                    8.59127498785852844e-02 -9.76246533925024473e-01 2.59241207829399986e-03
                    2.00824560029427057e-01 -9.60848391618718489e-01 2.44547418725300000e-03
                    3.14909131134665621e-01 -9.31867425680548589e-01 2.22077428728399999e-03
                    -2.42154937925004726e-01 -9.54209648055068915e-01 1.89187832491400005e-03
                    4.27478528173224070e-01 -8.87895688944866812e-01 2.00791878619099992e-03
                    -4.50733616169854578e-01 -8.78678625692422943e-01 1.74919241129499990e-03
                    -3.37292881327725114e-01 -9.31720524583063758e-01 1.38941196409599998e-03
                    -9.82350154705346063e-02 -9.91223466065941694e-01 1.13248117881799997e-03
                    1.56718632600337041e-02 -9.95996747980394947e-01 1.14561860890600000e-03
                    1.31139900271955490e-01 -9.87622561513300301e-01 1.10982691108799992e-03
                    2.46706557876677585e-01 -9.65659904398402325e-01 1.01516445954099991e-03
                    -2.07655398253820123e-01 -9.74852435145912910e-01 9.18301477314999973e-04
                    3.60690428367306026e-01 -9.29486468565670410e-01 9.19438803836000003e-04
                    4.69923038748497113e-01 -8.79707378419272823e-01 8.14130778808999970e-04
                    -4.22073245830824451e-01 -9.04135651352802716e-01 6.74237438331999985e-04
                    -3.10457265511967895e-01 -9.49186404329013356e-01 4.41837214846999991e-04
                    6.52371859800334186e-02 -1.12994120660238118e-01 1.46596687024809994e-02
                    1.94571594321364505e-01 -1.12424572060634595e-01 1.43293711259989999e-02
                    1.29247771730558014e-01 -2.24145363811504600e-01 1.41657314044679999e-02
                    2.56359546657133008e-01 -2.22347924443301787e-01 1.36793777585130002e-02
                    3.20621922583718633e-01 -1.11369892625493322e-01 1.36800980112599998e-02
                    1.90929811020464857e-01 -3.31542633276775744e-01 1.33610901887220006e-02
                    3.79227245030271332e-01 -2.19327117080325040e-01 1.28865763702140004e-02
                    3.14837316323372696e-01 -3.27707500025857357e-01 1.27309586541619994e-02
                    4.41274366132454354e-01 -1.09761572679574065e-01 1.27335707412020004e-02
                    2.49373612369882813e-01 -4.33161170177593813e-01 1.22728977091429992e-02
                    4.33751197464270211e-01 -3.21684818235892622e-01 1.18104934192140004e-02
                    4.95894235684805063e-01 -2.14872028798018810e-01 1.18119862010690000e-02
                    3.69292072583707887e-01 -4.26200971915065729e-01 1.15169048960539998e-02
                    5.54452436633114387e-01 -1.07261488677174444e-01 1.15199313878869992e-02
                    3.03993103342766025e-01 -5.26768718038455486e-01 1.09395418124159996e-02
                    5.46111311672082822e-01 -3.12742145416031569e-01 1.06252907191070004e-02
                    4.83993003433959346e-01 -4.15442792135906624e-01 1.04907059906140000e-02
                    6.04604604204736717e-01 -2.08171163750297233e-01 1.04828891908719992e-02
                    4.19639254970436859e-01 -5.14887602392660959e-01 1.00796115836390004e-02
                    6.58225901544846614e-01 -1.02892575444226669e-01 1.00794828416590002e-02
                    3.55118576816377207e-01 -6.09505446729273870e-01 9.40845773050300038e-03
                    5.92793358863897279e-01 -3.98892146500893108e-01 9.24078299032900057e-03
                    6.50942587188926836e-01 -2.98653239649445346e-01 9.20593261322900057e-03
                    5.30877780848832970e-01 -4.96356540085595765e-01 8.98950156821000032e-03
                    7.04144625226585164e-01 -1.96911249733332738e-01 8.93661852393799923e-03
                    4.67132549919417162e-01 -5.89646565876768669e-01 8.46634636712599933e-03
                    7.50923165944653248e-01 -9.37098677146950693e-02 8.46188329973500061e-03
                    6.94609545058963218e-01 -3.74144066198039138e-01 7.88432596683500027e-03
                    4.05359380480673770e-01 -6.77078978530161857e-01 7.72106443703700004e-03
                    6.36489593892303196e-01 -4.70233579648803912e-01 7.80084606203599977e-03
                    7.47488374440001935e-01 -2.75764000248896246e-01 7.67590729036300018e-03
                    5.74769992582348177e-01 -5.61970010543340281e-01 7.37352560194999996e-03
                    7.93625549141745212e-01 -1.75877743123834390e-01 7.28199558876000026e-03
                    5.13126999311486198e-01 -6.47700871504743536e-01 6.67061502106700011e-03
                    8.30830154268360599e-01 -7.20961092724231173e-02 6.75288352783299985e-03
                    7.23302415312991553e-01 -4.53526409726521418e-01 6.52417810093700035e-03
                    7.77369799560253427e-01 -3.58247201271682081e-01 6.47039073583199984e-03
                    6.64159749723096660e-01 -5.44472667196584603e-01 6.23376213407899997e-03
                    4.60599274129230130e-01 -7.24978779818310048e-01 6.02765717185200015e-03
                    8.24849972502917694e-01 -2.60472368706051283e-01 6.19955603562300037e-03
                    6.03037435184724746e-01 -6.27659789696723203e-01 5.38752222495999990e-03
                    8.63547605445974531e-01 -1.58435241606476818e-01 5.80751036954100033e-03
                    5.55876553003419538e-01 -6.97747919907965364e-01 4.53486784658799994e-03
                    8.91684001588963215e-01 -5.07357069956552581e-02 5.26685028270500004e-03
                    7.96998097459915122e-01 -4.34542834656064636e-01 5.16661701956700001e-03
                    7.40858255581746672e-01 -5.26343258500709510e-01 5.15119225902399992e-03
                    8.46224861884197876e-01 -3.39568989015640366e-01 5.01024438852399993e-03
                    5.02827413767285236e-01 -7.62205243653656717e-01 4.13473702879899995e-03
                    6.77584324774495483e-01 -6.13452607500896918e-01 4.81739668099000033e-03
                    8.87139502819643266e-01 -2.39997884164358216e-01 4.74850298747600021e-03
                    9.17857496194045552e-01 -1.34008708135026161e-01 4.40448832080200022e-03
                    6.20634893418241629e-01 -6.93245451409694291e-01 4.18053394324000016e-03
                    9.37800788228237980e-01 -2.46074562016906466e-02 3.78032759303999991e-03
                    5.57760155482306064e-01 -7.66367990202672322e-01 3.55846386921100004e-03
                    7.92495911555535115e-01 -5.23885315394653550e-01 3.85502295556000010e-03
                    8.47165983566849579e-01 -4.30541175975967583e-01 3.90916037461699981e-03
                    7.29768172986909325e-01 -6.11531275639734306e-01 3.69682193337999990e-03
                    8.94304328575331819e-01 -3.30504923347857860e-01 3.79249084190899990e-03
                    9.31419538925725932e-01 -2.23247385013044752e-01 3.54570802098299991e-03
                    6.68179560282742169e-01 -6.92886765652943937e-01 3.09316883568099997e-03
                    9.56526886688653266e-01 -1.10675125242810965e-01 3.17774833646300003e-03
                    4.92779607427050625e-01 -8.32824583288461984e-01 2.87756055216200008e-03
                    6.00338309245478374e-01 -7.67105602542828890e-01 2.49034745475699981e-03
                    8.34317587607139477e-01 -5.13053754917097260e-01 2.60136723227800005e-03
                    7.71100375680444983e-01 -6.04738843333724629e-01 2.48857966320900018e-03
                    8.88410673674870477e-01 -4.13720643058679227e-01 2.59241207829399986e-03
                    9.32531396341942598e-01 -3.06505025120042796e-01 2.44547418725300000e-03
                    9.64475429165895237e-01 -1.93214405393969030e-01 2.22077428728399999e-03
                    7.05292326789395463e-01 -6.86817151922432645e-01 1.89187832491400005e-03
                    9.82679486623552534e-01 -7.37405795020398269e-02 2.00791878619099992e-03
                    5.35591203527108650e-01 -8.29686074788930084e-01 1.74919241129499990e-03
                    6.38247202852433970e-01 -7.57964466036991968e-01 1.38941196409599998e-03
                    8.09307194705100486e-01 -5.80685751971611430e-01 1.13248117881799997e-03
                    8.70394417467726056e-01 -4.84426142282372629e-01 1.14561860890600000e-03
                    9.20876177757155134e-01 -3.80240795671379250e-01 1.10982691108799992e-03
                    9.59639287563407550e-01 -2.69175805797782774e-01 1.01516445954099991e-03
                    7.40419274650572223e-01 -6.67261067693739718e-01 9.18301477314999973e-04
                    9.85304108435409609e-01 -1.52376160414857159e-01 9.19438803836000003e-04
                    9.96810456981949256e-01 -3.28883998298591662e-02 8.14130778808999970e-04
                    5.71967819623304896e-01 -8.17593978823649925e-01 6.74237438331999985e-04
                    6.66790906319749133e-01 -7.43457080887321653e-01 4.41837214846999991e-04
                    ];

            case 2

                xyw=[
                    0.00000000000000000e+00 -0.00000000000000000e+00 1.48231258515199998e-02
                    1.12837432960665868e-01 -6.53099698497229958e-02 1.46550052854469998e-02
                    -1.35089747617184710e-14 -1.30633128478431004e-01 1.46601289931080001e-02
                    2.24375396355257994e-01 -0.00000000000000000e+00 1.43161642910620002e-02
                    1.12342388797948597e-01 -1.94777817298368877e-01 1.43318420700959993e-02
                    2.23875237151412859e-01 -1.29421201339195657e-01 1.41564828142480004e-02
                    -2.67789049204596934e-14 -2.58954671889630006e-01 1.41757432451159997e-02
                    3.32832869785287477e-01 -6.40789157248252833e-02 1.36549034840389993e-02
                    2.22356778580834108e-01 -2.56582184210307207e-01 1.36819770735700005e-02
                    1.11333487803831968e-01 -3.20810038840956824e-01 1.36948039982800005e-02
                    3.31286644624405768e-01 -1.91139528167454220e-01 1.33442999912940004e-02
                    -3.95753169825989000e-14 -3.82697248248083988e-01 1.33844682856060007e-02
                    4.37478323057291008e-01 -0.00000000000000000e+00 1.28439840133620000e-02
                    2.19795815804995798e-01 -3.79242201694888592e-01 1.29068157075800001e-02
                    4.36410144307665282e-01 -1.25683627935622877e-01 1.26923868994570001e-02
                    1.09804465871229509e-01 -4.41172624413793790e-01 1.27636747309160005e-02
                    3.28178035949764579e-01 -3.14902236277643632e-01 1.27331079005089994e-02
                    -5.16702925083657911e-14 -4.99656863590498990e-01 1.23124810284479993e-02
                    4.33220933250632934e-01 -2.49255436616029169e-01 1.22443415275810003e-02
                    5.36062621117270544e-01 -6.15112032146195239e-02 1.17504173106120005e-02
                    2.16230808933424445e-01 -4.95191104065842491e-01 1.18552725255409996e-02
                    3.23529450740014435e-01 -4.33158187731571664e-01 1.18411306953540001e-02
                    5.33284237574229381e-01 -1.83527148050910860e-01 1.14569635458349993e-02
                    1.07811213462251296e-01 -5.53684727034974178e-01 1.15700474175849995e-02
                    4.27969556469178725e-01 -3.68556696942499873e-01 1.15188349970590000e-02
                    -6.28413175180848254e-14 -6.07681785619797021e-01 1.09993260372280004e-02
                    5.27880828851488126e-01 -3.02470133207228653e-01 1.08916599392419999e-02
                    6.28576433103623988e-01 -0.00000000000000000e+00 1.05572378623799996e-02
                    3.17382414995297846e-01 -5.43759082660381421e-01 1.06933389262469994e-02
                    6.26802302110957221e-01 -1.19129741387730803e-01 1.04086677071329996e-02
                    2.11780486927457184e-01 -6.02286206045308470e-01 1.05580329937140002e-02
                    4.20674557734878829e-01 -4.81459799360681007e-01 1.05415604265690008e-02
                    1.05494659158648238e-01 -6.56221922673913993e-01 1.01554736102849995e-02
                    6.21694666807976692e-01 -2.36254985457061362e-01 9.98460125115700065e-03
                    5.19980533812632650e-01 -4.16255510265814588e-01 1.00888329474420006e-02
                    -7.28699030894243711e-14 -7.04659204743429024e-01 9.49430123143600070e-03
                    6.13669047860877059e-01 -3.49321474877237192e-01 9.34070311505899933e-03
                    7.12227969912828551e-01 -5.76095975982448619e-02 9.16844199076500069e-03
                    3.09065068009761246e-01 -6.45071554139860504e-01 9.29848135252200024e-03
                    4.10362096875270421e-01 -5.86508613903598874e-01 9.32351072820499990e-03
                    7.07941339095839561e-01 -1.71729669336721180e-01 8.86059320166899915e-03
                    2.06136158326821972e-01 -6.98744998347219370e-01 9.03387615400600037e-03
                    5.08484362141080615e-01 -5.23608285482623725e-01 9.06784470665199932e-03
                    1.02731819240258718e-01 -7.46881144619204740e-01 8.55591190669499971e-03
                    7.00181204791145761e-01 -2.82518028254914177e-01 8.33102004526299969e-03
                    6.01758767718018284e-01 -4.57233916163265475e-01 8.51642807162399963e-03
                    7.87951800544569037e-01 -0.00000000000000000e+00 7.85184400079200033e-03
                    -8.15611207287998588e-14 -7.88704143056284002e-01 7.83166322365400069e-03
                    3.91487303480710702e-01 -6.84875370201805000e-01 7.88112260952500074e-03
                    6.88514891010830188e-01 -3.88964098337549469e-01 7.67894908294199964e-03
                    7.85209296345330099e-01 -1.11183005024141524e-01 7.65837547845699997e-03
                    4.87844461262681162e-01 -6.26106709232768566e-01 7.86487166401299981e-03
                    2.93621303683378687e-01 -7.37702220466245229e-01 7.63034736577799990e-03
                    5.80578341798693387e-01 -5.62043167569809765e-01 7.53216769862400000e-03
                    7.77994878477669460e-01 -2.19195341205039390e-01 7.16274450921400036e-03
                    1.95528110087736129e-01 -7.84419959952223023e-01 7.23749359942999979e-03
                    6.67562406662079444e-01 -4.94058867584967198e-01 6.88873705365400033e-03
                    9.70065467489010097e-02 -8.24824484789229651e-01 6.73787423472200012e-03
                    7.67029080112698991e-01 -3.22253572954260048e-01 6.58760147645800001e-03
                    8.52134631062725467e-01 -5.53572856340550645e-02 6.56397386974100043e-03
                    4.59034776845581083e-01 -7.20502236320693323e-01 6.75929695408600002e-03
                    3.60476062053627755e-01 -7.75748502109505522e-01 6.62734020038700019e-03
                    -8.88433355383826385e-14 -8.59123883977226055e-01 5.95498878444399981e-03
                    7.46614051161292314e-01 -4.25129492237797779e-01 5.93418824640699980e-03
                    5.54058683467657187e-01 -6.58521549423318753e-01 6.56207460755499957e-03
                    8.45686112457160299e-01 -1.63577885739557866e-01 6.11001174754800033e-03
                    2.61474565558392047e-01 -8.23410980567719109e-01 6.14286717416199970e-03
                    6.43295885266826240e-01 -5.90899505408985060e-01 6.07294624840599984e-03
                    8.35299944248545145e-01 -2.64651765964598051e-01 5.42167623554899985e-03
                    1.65694454853941825e-01 -8.62333888244206537e-01 5.37501544402300014e-03
                    8.14861365182181219e-01 -3.62422812612702150e-01 4.59568821999400013e-03
                    7.24916525851842852e-01 -5.19796951937957386e-01 5.33618992961100002e-03
                    7.57696286253324719e-02 -8.92607674214299651e-01 4.56993737540900037e-03
                    9.07054213034169998e-01 -0.00000000000000000e+00 5.46241231307799998e-03
                    4.42242634627807385e-01 -7.92761734691979081e-01 5.44663241367300023e-03
                    9.02682415047762921e-01 -1.12551640683923038e-01 5.27442327957799995e-03
                    5.38503350869324771e-01 -7.33410787171000833e-01 5.35090579030399974e-03
                    3.41883918511743934e-01 -8.44245053680024005e-01 5.27660095420900022e-03
                    7.97263860238179078e-01 -4.48734093482190621e-01 4.30018157389400017e-03
                    6.28891758057308015e-01 -6.66846417008255976e-01 4.99824430716199961e-03
                    8.91410221869499919e-01 -2.20462665262784380e-01 4.69853362942899977e-03
                    2.39303273920731613e-01 -8.87478489699914386e-01 4.85981655752300030e-03
                    -9.52300995612054705e-14 -9.20884526799590986e-01 3.73536796695399980e-03
                    8.67733073743958361e-01 -3.26426035545693716e-01 3.92960116374199957e-03
                    7.11727954675998586e-01 -5.94936671603206291e-01 4.42335154098000023e-03
                    1.37622048239333661e-01 -9.21395318784136386e-01 4.15821285139300004e-03
                    7.85705451047371217e-01 -5.19485604669765788e-01 3.62831066641600012e-03
                    8.54923285223178286e-01 -4.09412039322762766e-01 3.54980989056999981e-03
                    9.47934243590281445e-01 -5.66124354624876133e-02 4.04533690231499970e-03
                    5.10701619053915146e-02 -9.48461845643240653e-01 2.74997177325300018e-03
                    4.31400732085064409e-01 -8.46502493858650507e-01 3.96259596692300018e-03
                    5.27044895246795542e-01 -7.90846040296282915e-01 3.95027933900699967e-03
                    3.31190242097535892e-01 -8.93371118167587630e-01 3.78389000269900013e-03
                    9.38336052632894435e-01 -1.67229564939150643e-01 3.73662685985800018e-03
                    6.17099306322381747e-01 -7.26599206167858447e-01 3.74011918994900019e-03
                    9.14313291474611156e-01 -2.79786343589928654e-01 3.24965209930600006e-03
                    2.27512191539795045e-01 -9.31494825820701089e-01 3.42690501398499985e-03
                    7.00088454217311917e-01 -6.55272608849804983e-01 3.33863712365500007e-03
                    7.74610359381933566e-01 -5.78055097179581034e-01 2.75742228760000006e-03
                    1.24207053402499809e-01 -9.59457065285587496e-01 2.61632989023800021e-03
                    8.43237196597550698e-01 -4.78498653963401688e-01 2.71174567446600019e-03
                    9.03729154890727804e-01 -3.60847382789604543e-01 2.63376060625700008e-03
                    -1.00839758056489126e-13 -9.75130481941289040e-01 2.61465978004600010e-03
                    9.78840080593204043e-01 -0.00000000000000000e+00 2.65105827225200000e-03
                    9.52071689063728854e-01 -2.30081967668126636e-01 2.36415247803799982e-03
                    5.25226236432522087e-01 -8.26898477180594393e-01 2.47171238457499992e-03
                    4.30757069856774533e-01 -8.79854730654636819e-01 2.51474341781900016e-03
                    9.73483764641486338e-01 -1.13088218146317851e-01 2.56557226477100017e-03
                    6.12063267272664469e-01 -7.66053023865866933e-01 2.31192320871599979e-03
                    3.29630732449447017e-01 -9.23991279910994168e-01 2.39814690487599996e-03
                    6.91255236876849422e-01 -6.98473608830743165e-01 2.06148693511099988e-03
                    2.22977164564219926e-01 -9.58713847167306232e-01 2.10424687186700019e-03
                    7.62788131249490298e-01 -6.24155041982559666e-01 1.72521161714499997e-03
                    1.05994309894318781e-01 -9.80967222184524035e-01 1.70640551365699999e-03
                    8.28829959863797372e-01 -5.35834627117448226e-01 1.69321452361100000e-03
                    8.88701635210429663e-01 -4.31501747167771899e-01 1.68331112853099991e-03
                    9.39853926911746629e-01 -3.12587613527860464e-01 1.54299866912100000e-03
                    -1.02688211514191922e-13 -9.93005210578044029e-01 1.30079144601600010e-03
                    9.77851086434011751e-01 -1.82993407168900141e-01 1.18604652062799997e-03
                    9.94313150889492681e-01 -5.61190736687032093e-02 1.15476896378499997e-03
                    5.37787545783237286e-01 -8.38476863760304392e-01 1.12924676460900001e-03
                    4.39228604881380602e-01 -8.94056747347730663e-01 1.12198085237100002e-03
                    6.30028132364980009e-01 -7.71845513335237765e-01 1.07453738060199998e-03
                    3.36063190586354477e-01 -9.38012850425386180e-01 1.02961909163399993e-03
                    7.15075772267507626e-01 -6.94574875831306948e-01 9.34856278170000023e-04
                    1.31524020318368878e-01 -9.88270222271415144e-01 6.98243074158999989e-04
                    2.31420100207260448e-01 -9.69905039743745090e-01 7.97390131281999954e-04
                    7.92689859343882031e-01 -6.05302780969197318e-01 8.10401076300000045e-04
                    8.60358204121903003e-01 -5.04872205483959058e-01 7.63219634713999985e-04
                    9.15903427116124380e-01 -3.95900950922521533e-01 6.92280023180999959e-04
                    9.59291700177966278e-01 -2.78024641686341167e-01 4.58817188194999984e-04
                    9.89547932751443038e-01 -1.38330978668465887e-01 1.75322281775999993e-04
                    5.27738387936431200e-02 -9.97940203891528710e-01 2.48540944503000010e-04
                    1.12837432960665868e-01 6.53099698497229958e-02 1.46550052854469998e-02
                    -1.35089747617184710e-14 1.30633128478431004e-01 1.46601289931080001e-02
                    1.12342388797948597e-01 1.94777817298368877e-01 1.43318420700959993e-02
                    2.23875237151412859e-01 1.29421201339195657e-01 1.41564828142480004e-02
                    -2.67789049204596934e-14 2.58954671889630006e-01 1.41757432451159997e-02
                    3.32832869785287477e-01 6.40789157248252833e-02 1.36549034840389993e-02
                    2.22356778580834108e-01 2.56582184210307207e-01 1.36819770735700005e-02
                    1.11333487803831968e-01 3.20810038840956824e-01 1.36948039982800005e-02
                    3.31286644624405768e-01 1.91139528167454220e-01 1.33442999912940004e-02
                    -3.95753169825989000e-14 3.82697248248083988e-01 1.33844682856060007e-02
                    2.19795815804995798e-01 3.79242201694888592e-01 1.29068157075800001e-02
                    4.36410144307665282e-01 1.25683627935622877e-01 1.26923868994570001e-02
                    1.09804465871229509e-01 4.41172624413793790e-01 1.27636747309160005e-02
                    3.28178035949764579e-01 3.14902236277643632e-01 1.27331079005089994e-02
                    -5.16702925083657911e-14 4.99656863590498990e-01 1.23124810284479993e-02
                    4.33220933250632934e-01 2.49255436616029169e-01 1.22443415275810003e-02
                    5.36062621117270544e-01 6.15112032146195239e-02 1.17504173106120005e-02
                    2.16230808933424445e-01 4.95191104065842491e-01 1.18552725255409996e-02
                    3.23529450740014435e-01 4.33158187731571664e-01 1.18411306953540001e-02
                    5.33284237574229381e-01 1.83527148050910860e-01 1.14569635458349993e-02
                    1.07811213462251296e-01 5.53684727034974178e-01 1.15700474175849995e-02
                    4.27969556469178725e-01 3.68556696942499873e-01 1.15188349970590000e-02
                    -6.28413175180848254e-14 6.07681785619797021e-01 1.09993260372280004e-02
                    5.27880828851488126e-01 3.02470133207228653e-01 1.08916599392419999e-02
                    3.17382414995297846e-01 5.43759082660381421e-01 1.06933389262469994e-02
                    6.26802302110957221e-01 1.19129741387730803e-01 1.04086677071329996e-02
                    2.11780486927457184e-01 6.02286206045308470e-01 1.05580329937140002e-02
                    4.20674557734878829e-01 4.81459799360681007e-01 1.05415604265690008e-02
                    1.05494659158648238e-01 6.56221922673913993e-01 1.01554736102849995e-02
                    6.21694666807976692e-01 2.36254985457061362e-01 9.98460125115700065e-03
                    5.19980533812632650e-01 4.16255510265814588e-01 1.00888329474420006e-02
                    -7.28699030894243711e-14 7.04659204743429024e-01 9.49430123143600070e-03
                    6.13669047860877059e-01 3.49321474877237192e-01 9.34070311505899933e-03
                    7.12227969912828551e-01 5.76095975982448619e-02 9.16844199076500069e-03
                    3.09065068009761246e-01 6.45071554139860504e-01 9.29848135252200024e-03
                    4.10362096875270421e-01 5.86508613903598874e-01 9.32351072820499990e-03
                    7.07941339095839561e-01 1.71729669336721180e-01 8.86059320166899915e-03
                    2.06136158326821972e-01 6.98744998347219370e-01 9.03387615400600037e-03
                    5.08484362141080615e-01 5.23608285482623725e-01 9.06784470665199932e-03
                    1.02731819240258718e-01 7.46881144619204740e-01 8.55591190669499971e-03
                    7.00181204791145761e-01 2.82518028254914177e-01 8.33102004526299969e-03
                    6.01758767718018284e-01 4.57233916163265475e-01 8.51642807162399963e-03
                    -8.15611207287998588e-14 7.88704143056284002e-01 7.83166322365400069e-03
                    3.91487303480710702e-01 6.84875370201805000e-01 7.88112260952500074e-03
                    6.88514891010830188e-01 3.88964098337549469e-01 7.67894908294199964e-03
                    7.85209296345330099e-01 1.11183005024141524e-01 7.65837547845699997e-03
                    4.87844461262681162e-01 6.26106709232768566e-01 7.86487166401299981e-03
                    2.93621303683378687e-01 7.37702220466245229e-01 7.63034736577799990e-03
                    5.80578341798693387e-01 5.62043167569809765e-01 7.53216769862400000e-03
                    7.77994878477669460e-01 2.19195341205039390e-01 7.16274450921400036e-03
                    1.95528110087736129e-01 7.84419959952223023e-01 7.23749359942999979e-03
                    6.67562406662079444e-01 4.94058867584967198e-01 6.88873705365400033e-03
                    9.70065467489010097e-02 8.24824484789229651e-01 6.73787423472200012e-03
                    7.67029080112698991e-01 3.22253572954260048e-01 6.58760147645800001e-03
                    8.52134631062725467e-01 5.53572856340550645e-02 6.56397386974100043e-03
                    4.59034776845581083e-01 7.20502236320693323e-01 6.75929695408600002e-03
                    3.60476062053627755e-01 7.75748502109505522e-01 6.62734020038700019e-03
                    -8.88433355383826385e-14 8.59123883977226055e-01 5.95498878444399981e-03
                    7.46614051161292314e-01 4.25129492237797779e-01 5.93418824640699980e-03
                    5.54058683467657187e-01 6.58521549423318753e-01 6.56207460755499957e-03
                    8.45686112457160299e-01 1.63577885739557866e-01 6.11001174754800033e-03
                    2.61474565558392047e-01 8.23410980567719109e-01 6.14286717416199970e-03
                    6.43295885266826240e-01 5.90899505408985060e-01 6.07294624840599984e-03
                    8.35299944248545145e-01 2.64651765964598051e-01 5.42167623554899985e-03
                    1.65694454853941825e-01 8.62333888244206537e-01 5.37501544402300014e-03
                    8.14861365182181219e-01 3.62422812612702150e-01 4.59568821999400013e-03
                    7.24916525851842852e-01 5.19796951937957386e-01 5.33618992961100002e-03
                    7.57696286253324719e-02 8.92607674214299651e-01 4.56993737540900037e-03
                    4.42242634627807385e-01 7.92761734691979081e-01 5.44663241367300023e-03
                    9.02682415047762921e-01 1.12551640683923038e-01 5.27442327957799995e-03
                    5.38503350869324771e-01 7.33410787171000833e-01 5.35090579030399974e-03
                    3.41883918511743934e-01 8.44245053680024005e-01 5.27660095420900022e-03
                    7.97263860238179078e-01 4.48734093482190621e-01 4.30018157389400017e-03
                    6.28891758057308015e-01 6.66846417008255976e-01 4.99824430716199961e-03
                    8.91410221869499919e-01 2.20462665262784380e-01 4.69853362942899977e-03
                    2.39303273920731613e-01 8.87478489699914386e-01 4.85981655752300030e-03
                    -9.52300995612054705e-14 9.20884526799590986e-01 3.73536796695399980e-03
                    8.67733073743958361e-01 3.26426035545693716e-01 3.92960116374199957e-03
                    7.11727954675998586e-01 5.94936671603206291e-01 4.42335154098000023e-03
                    1.37622048239333661e-01 9.21395318784136386e-01 4.15821285139300004e-03
                    7.85705451047371217e-01 5.19485604669765788e-01 3.62831066641600012e-03
                    8.54923285223178286e-01 4.09412039322762766e-01 3.54980989056999981e-03
                    9.47934243590281445e-01 5.66124354624876133e-02 4.04533690231499970e-03
                    5.10701619053915146e-02 9.48461845643240653e-01 2.74997177325300018e-03
                    4.31400732085064409e-01 8.46502493858650507e-01 3.96259596692300018e-03
                    5.27044895246795542e-01 7.90846040296282915e-01 3.95027933900699967e-03
                    3.31190242097535892e-01 8.93371118167587630e-01 3.78389000269900013e-03
                    9.38336052632894435e-01 1.67229564939150643e-01 3.73662685985800018e-03
                    6.17099306322381747e-01 7.26599206167858447e-01 3.74011918994900019e-03
                    9.14313291474611156e-01 2.79786343589928654e-01 3.24965209930600006e-03
                    2.27512191539795045e-01 9.31494825820701089e-01 3.42690501398499985e-03
                    7.00088454217311917e-01 6.55272608849804983e-01 3.33863712365500007e-03
                    7.74610359381933566e-01 5.78055097179581034e-01 2.75742228760000006e-03
                    1.24207053402499809e-01 9.59457065285587496e-01 2.61632989023800021e-03
                    8.43237196597550698e-01 4.78498653963401688e-01 2.71174567446600019e-03
                    9.03729154890727804e-01 3.60847382789604543e-01 2.63376060625700008e-03
                    -1.00839758056489126e-13 9.75130481941289040e-01 2.61465978004600010e-03
                    9.52071689063728854e-01 2.30081967668126636e-01 2.36415247803799982e-03
                    5.25226236432522087e-01 8.26898477180594393e-01 2.47171238457499992e-03
                    4.30757069856774533e-01 8.79854730654636819e-01 2.51474341781900016e-03
                    9.73483764641486338e-01 1.13088218146317851e-01 2.56557226477100017e-03
                    6.12063267272664469e-01 7.66053023865866933e-01 2.31192320871599979e-03
                    3.29630732449447017e-01 9.23991279910994168e-01 2.39814690487599996e-03
                    6.91255236876849422e-01 6.98473608830743165e-01 2.06148693511099988e-03
                    2.22977164564219926e-01 9.58713847167306232e-01 2.10424687186700019e-03
                    7.62788131249490298e-01 6.24155041982559666e-01 1.72521161714499997e-03
                    1.05994309894318781e-01 9.80967222184524035e-01 1.70640551365699999e-03
                    8.28829959863797372e-01 5.35834627117448226e-01 1.69321452361100000e-03
                    8.88701635210429663e-01 4.31501747167771899e-01 1.68331112853099991e-03
                    9.39853926911746629e-01 3.12587613527860464e-01 1.54299866912100000e-03
                    -1.02688211514191922e-13 9.93005210578044029e-01 1.30079144601600010e-03
                    9.77851086434011751e-01 1.82993407168900141e-01 1.18604652062799997e-03
                    9.94313150889492681e-01 5.61190736687032093e-02 1.15476896378499997e-03
                    5.37787545783237286e-01 8.38476863760304392e-01 1.12924676460900001e-03
                    4.39228604881380602e-01 8.94056747347730663e-01 1.12198085237100002e-03
                    6.30028132364980009e-01 7.71845513335237765e-01 1.07453738060199998e-03
                    3.36063190586354477e-01 9.38012850425386180e-01 1.02961909163399993e-03
                    7.15075772267507626e-01 6.94574875831306948e-01 9.34856278170000023e-04
                    1.31524020318368878e-01 9.88270222271415144e-01 6.98243074158999989e-04
                    2.31420100207260448e-01 9.69905039743745090e-01 7.97390131281999954e-04
                    7.92689859343882031e-01 6.05302780969197318e-01 8.10401076300000045e-04
                    8.60358204121903003e-01 5.04872205483959058e-01 7.63219634713999985e-04
                    9.15903427116124380e-01 3.95900950922521533e-01 6.92280023180999959e-04
                    9.59291700177966278e-01 2.78024641686341167e-01 4.58817188194999984e-04
                    9.89547932751443038e-01 1.38330978668465887e-01 1.75322281775999993e-04
                    5.27738387936431200e-02 9.97940203891528710e-01 2.48540944503000010e-04
                    -1.12837432960665854e-01 6.53099698497230097e-02 1.46550052854469998e-02
                    -2.24375396355257994e-01 2.74780610953885406e-17 1.43161642910620002e-02
                    -1.12342388797948570e-01 1.94777817298368905e-01 1.43318420700959993e-02
                    -2.23875237151412831e-01 1.29421201339195685e-01 1.41564828142480004e-02
                    -3.32832869785287422e-01 6.40789157248253527e-02 1.36549034840389993e-02
                    -2.22356778580834025e-01 2.56582184210307263e-01 1.36819770735700005e-02
                    -1.11333487803831926e-01 3.20810038840956824e-01 1.36948039982800005e-02
                    -3.31286644624405713e-01 1.91139528167454248e-01 1.33442999912940004e-02
                    -4.37478323057291008e-01 5.35756428028463164e-17 1.28439840133620000e-02
                    -2.19795815804995742e-01 3.79242201694888648e-01 1.29068157075800001e-02
                    -4.36410144307665282e-01 1.25683627935622905e-01 1.26923868994570001e-02
                    -1.09804465871229454e-01 4.41172624413793846e-01 1.27636747309160005e-02
                    -3.28178035949764524e-01 3.14902236277643688e-01 1.27331079005089994e-02
                    -4.33220933250632989e-01 2.49255436616029141e-01 1.22443415275810003e-02
                    -5.36062621117270544e-01 6.15112032146196627e-02 1.17504173106120005e-02
                    -2.16230808933424390e-01 4.95191104065842547e-01 1.18552725255409996e-02
                    -3.23529450740014435e-01 4.33158187731571664e-01 1.18411306953540001e-02
                    -5.33284237574229381e-01 1.83527148050910943e-01 1.14569635458349993e-02
                    -1.07811213462251226e-01 5.53684727034974178e-01 1.15700474175849995e-02
                    -4.27969556469178725e-01 3.68556696942499873e-01 1.15188349970590000e-02
                    -5.27880828851488126e-01 3.02470133207228709e-01 1.08916599392419999e-02
                    -6.28576433103623988e-01 7.69784116819813515e-17 1.05572378623799996e-02
                    -3.17382414995297957e-01 5.43759082660381421e-01 1.06933389262469994e-02
                    -6.26802302110957110e-01 1.19129741387730900e-01 1.04086677071329996e-02
                    -2.11780486927457129e-01 6.02286206045308581e-01 1.05580329937140002e-02
                    -4.20674557734878773e-01 4.81459799360681062e-01 1.05415604265690008e-02
                    -1.05494659158648155e-01 6.56221922673913993e-01 1.01554736102849995e-02
                    -6.21694666807976692e-01 2.36254985457061473e-01 9.98460125115700065e-03
                    -5.19980533812632428e-01 4.16255510265814699e-01 1.00888329474420006e-02
                    -6.13669047860876948e-01 3.49321474877237359e-01 9.34070311505899933e-03
                    -7.12227969912828551e-01 5.76095975982449521e-02 9.16844199076500069e-03
                    -3.09065068009761301e-01 6.45071554139860392e-01 9.29848135252200024e-03
                    -4.10362096875270255e-01 5.86508613903599096e-01 9.32351072820499990e-03
                    -7.07941339095839561e-01 1.71729669336721291e-01 8.86059320166899915e-03
                    -2.06136158326821889e-01 6.98744998347219481e-01 9.03387615400600037e-03
                    -5.08484362141080504e-01 5.23608285482623836e-01 9.06784470665199932e-03
                    -1.02731819240258621e-01 7.46881144619204740e-01 8.55591190669499971e-03
                    -7.00181204791145650e-01 2.82518028254914400e-01 8.33102004526299969e-03
                    -6.01758767718018173e-01 4.57233916163265586e-01 8.51642807162399963e-03
                    -7.87951800544569037e-01 9.64962650419300127e-17 7.85184400079200033e-03
                    -3.91487303480710425e-01 6.84875370201805223e-01 7.88112260952500074e-03
                    -6.88514891010830188e-01 3.88964098337549358e-01 7.67894908294199964e-03
                    -7.85209296345330099e-01 1.11183005024141787e-01 7.65837547845699997e-03
                    -4.87844461262681051e-01 6.26106709232768566e-01 7.86487166401299981e-03
                    -2.93621303683378576e-01 7.37702220466245229e-01 7.63034736577799990e-03
                    -5.80578341798693165e-01 5.62043167569809876e-01 7.53216769862400000e-03
                    -7.77994878477669460e-01 2.19195341205039557e-01 7.16274450921400036e-03
                    -1.95528110087736046e-01 7.84419959952223023e-01 7.23749359942999979e-03
                    -6.67562406662079444e-01 4.94058867584967198e-01 6.88873705365400033e-03
                    -9.70065467489008987e-02 8.24824484789229651e-01 6.73787423472200012e-03
                    -7.67029080112698880e-01 3.22253572954260159e-01 6.58760147645800001e-03
                    -8.52134631062725467e-01 5.53572856340553282e-02 6.56397386974100043e-03
                    -4.59034776845580972e-01 7.20502236320693434e-01 6.75929695408600002e-03
                    -3.60476062053627699e-01 7.75748502109505522e-01 6.62734020038700019e-03
                    -7.46614051161292314e-01 4.25129492237797835e-01 5.93418824640699980e-03
                    -5.54058683467657076e-01 6.58521549423318864e-01 6.56207460755499957e-03
                    -8.45686112457160299e-01 1.63577885739557838e-01 6.11001174754800033e-03
                    -2.61474565558391936e-01 8.23410980567719220e-01 6.14286717416199970e-03
                    -6.43295885266826240e-01 5.90899505408985060e-01 6.07294624840599984e-03
                    -8.35299944248545034e-01 2.64651765964598162e-01 5.42167623554899985e-03
                    -1.65694454853941714e-01 8.62333888244206537e-01 5.37501544402300014e-03
                    -8.14861365182181219e-01 3.62422812612702205e-01 4.59568821999400013e-03
                    -7.24916525851842741e-01 5.19796951937957497e-01 5.33618992961100002e-03
                    -7.57696286253323609e-02 8.92607674214299651e-01 4.56993737540900037e-03
                    -9.07054213034169998e-01 1.11082103864541778e-16 5.46241231307799998e-03
                    -4.42242634627807440e-01 7.92761734691979081e-01 5.44663241367300023e-03
                    -9.02682415047762921e-01 1.12551640683923218e-01 5.27442327957799995e-03
                    -5.38503350869324882e-01 7.33410787171000833e-01 5.35090579030399974e-03
                    -3.41883918511743823e-01 8.44245053680024005e-01 5.27660095420900022e-03
                    -7.97263860238179078e-01 4.48734093482190621e-01 4.30018157389400017e-03
                    -6.28891758057307904e-01 6.66846417008256198e-01 4.99824430716199961e-03
                    -8.91410221869499919e-01 2.20462665262784602e-01 4.69853362942899977e-03
                    -2.39303273920731502e-01 8.87478489699914386e-01 4.85981655752300030e-03
                    -8.67733073743958250e-01 3.26426035545693827e-01 3.92960116374199957e-03
                    -7.11727954675998586e-01 5.94936671603206180e-01 4.42335154098000023e-03
                    -1.37622048239333522e-01 9.21395318784136386e-01 4.15821285139300004e-03
                    -7.85705451047371106e-01 5.19485604669766010e-01 3.62831066641600012e-03
                    -8.54923285223178175e-01 4.09412039322762988e-01 3.54980989056999981e-03
                    -9.47934243590281445e-01 5.66124354624878909e-02 4.04533690231499970e-03
                    -5.10701619053913966e-02 9.48461845643240653e-01 2.74997177325300018e-03
                    -4.31400732085064353e-01 8.46502493858650618e-01 3.96259596692300018e-03
                    -5.27044895246795653e-01 7.90846040296282915e-01 3.95027933900699967e-03
                    -3.31190242097535781e-01 8.93371118167587630e-01 3.78389000269900013e-03
                    -9.38336052632894435e-01 1.67229564939150949e-01 3.73662685985800018e-03
                    -6.17099306322381858e-01 7.26599206167858447e-01 3.74011918994900019e-03
                    -9.14313291474611045e-01 2.79786343589928876e-01 3.24965209930600006e-03
                    -2.27512191539794933e-01 9.31494825820701200e-01 3.42690501398499985e-03
                    -7.00088454217311806e-01 6.55272608849805094e-01 3.33863712365500007e-03
                    -7.74610359381933566e-01 5.78055097179581034e-01 2.75742228760000006e-03
                    -1.24207053402499698e-01 9.59457065285587496e-01 2.61632989023800021e-03
                    -8.43237196597550587e-01 4.78498653963401910e-01 2.71174567446600019e-03
                    -9.03729154890727804e-01 3.60847382789604709e-01 2.63376060625700008e-03
                    -9.78840080593204043e-01 1.19873337157560465e-16 2.65105827225200000e-03
                    -9.52071689063728854e-01 2.30081967668126719e-01 2.36415247803799982e-03
                    -5.25226236432521865e-01 8.26898477180594615e-01 2.47171238457499992e-03
                    -4.30757069856774200e-01 8.79854730654636930e-01 2.51474341781900016e-03
                    -9.73483764641486338e-01 1.13088218146317920e-01 2.56557226477100017e-03
                    -6.12063267272664469e-01 7.66053023865866933e-01 2.31192320871599979e-03
                    -3.29630732449446906e-01 9.23991279910994168e-01 2.39814690487599996e-03
                    -6.91255236876849311e-01 6.98473608830743276e-01 2.06148693511099988e-03
                    -2.22977164564219787e-01 9.58713847167306232e-01 2.10424687186700019e-03
                    -7.62788131249490298e-01 6.24155041982559777e-01 1.72521161714499997e-03
                    -1.05994309894318670e-01 9.80967222184524035e-01 1.70640551365699999e-03
                    -8.28829959863797261e-01 5.35834627117448337e-01 1.69321452361100000e-03
                    -8.88701635210429663e-01 4.31501747167771954e-01 1.68331112853099991e-03
                    -9.39853926911746518e-01 3.12587613527860575e-01 1.54299866912100000e-03
                    -9.77851086434011862e-01 1.82993407168900057e-01 1.18604652062799997e-03
                    -9.94313150889492681e-01 5.61190736687035424e-02 1.15476896378499997e-03
                    -5.37787545783237397e-01 8.38476863760304281e-01 1.12924676460900001e-03
                    -4.39228604881380658e-01 8.94056747347730663e-01 1.12198085237100002e-03
                    -6.30028132364980120e-01 7.71845513335237765e-01 1.07453738060199998e-03
                    -3.36063190586354366e-01 9.38012850425386180e-01 1.02961909163399993e-03
                    -7.15075772267507626e-01 6.94574875831306837e-01 9.34856278170000023e-04
                    -1.31524020318368767e-01 9.88270222271415144e-01 6.98243074158999989e-04
                    -2.31420100207260310e-01 9.69905039743745201e-01 7.97390131281999954e-04
                    -7.92689859343882142e-01 6.05302780969197318e-01 8.10401076300000045e-04
                    -8.60358204121902892e-01 5.04872205483959169e-01 7.63219634713999985e-04
                    -9.15903427116124269e-01 3.95900950922521810e-01 6.92280023180999959e-04
                    -9.59291700177966389e-01 2.78024641686341112e-01 4.58817188194999984e-04
                    -9.89547932751443038e-01 1.38330978668465998e-01 1.75322281775999993e-04
                    -5.27738387936429951e-02 9.97940203891528710e-01 2.48540944503000010e-04
                    -1.12837432960665868e-01 -6.53099698497229819e-02 1.46550052854469998e-02
                    -1.12342388797948625e-01 -1.94777817298368877e-01 1.43318420700959993e-02
                    -2.23875237151412859e-01 -1.29421201339195630e-01 1.41564828142480004e-02
                    -3.32832869785287477e-01 -6.40789157248252694e-02 1.36549034840389993e-02
                    -2.22356778580834108e-01 -2.56582184210307207e-01 1.36819770735700005e-02
                    -1.11333487803832148e-01 -3.20810038840956713e-01 1.36948039982800005e-02
                    -3.31286644624405768e-01 -1.91139528167454165e-01 1.33442999912940004e-02
                    -2.19795815804996020e-01 -3.79242201694888481e-01 1.29068157075800001e-02
                    -4.36410144307665337e-01 -1.25683627935622794e-01 1.26923868994570001e-02
                    -1.09804465871229662e-01 -4.41172624413793790e-01 1.27636747309160005e-02
                    -3.28178035949764579e-01 -3.14902236277643632e-01 1.27331079005089994e-02
                    -4.33220933250633045e-01 -2.49255436616029030e-01 1.22443415275810003e-02
                    -5.36062621117270544e-01 -6.15112032146195239e-02 1.17504173106120005e-02
                    -2.16230808933424529e-01 -4.95191104065842491e-01 1.18552725255409996e-02
                    -3.23529450740014546e-01 -4.33158187731571609e-01 1.18411306953540001e-02
                    -5.33284237574229381e-01 -1.83527148050910777e-01 1.14569635458349993e-02
                    -1.07811213462251601e-01 -5.53684727034974067e-01 1.15700474175849995e-02
                    -4.27969556469178780e-01 -3.68556696942499762e-01 1.15188349970590000e-02
                    -5.27880828851488126e-01 -3.02470133207228542e-01 1.08916599392419999e-02
                    -3.17382414995298068e-01 -5.43759082660381421e-01 1.06933389262469994e-02
                    -6.26802302110957221e-01 -1.19129741387730748e-01 1.04086677071329996e-02
                    -2.11780486927457406e-01 -6.02286206045308470e-01 1.05580329937140002e-02
                    -4.20674557734878829e-01 -4.81459799360681007e-01 1.05415604265690008e-02
                    -1.05494659158648182e-01 -6.56221922673913993e-01 1.01554736102849995e-02
                    -6.21694666807976692e-01 -2.36254985457061334e-01 9.98460125115700065e-03
                    -5.19980533812632539e-01 -4.16255510265814588e-01 1.00888329474420006e-02
                    -6.13669047860877059e-01 -3.49321474877237192e-01 9.34070311505899933e-03
                    -7.12227969912828551e-01 -5.76095975982447717e-02 9.16844199076500069e-03
                    -3.09065068009761468e-01 -6.45071554139860392e-01 9.29848135252200024e-03
                    -4.10362096875270366e-01 -5.86508613903598985e-01 9.32351072820499990e-03
                    -7.07941339095839561e-01 -1.71729669336721125e-01 8.86059320166899915e-03
                    -2.06136158326821778e-01 -6.98744998347219481e-01 9.03387615400600037e-03
                    -5.08484362141080615e-01 -5.23608285482623725e-01 9.06784470665199932e-03
                    -1.02731819240258468e-01 -7.46881144619204740e-01 8.55591190669499971e-03
                    -7.00181204791145761e-01 -2.82518028254914233e-01 8.33102004526299969e-03
                    -6.01758767718018284e-01 -4.57233916163265475e-01 8.51642807162399963e-03
                    -3.91487303480710591e-01 -6.84875370201805111e-01 7.88112260952500074e-03
                    -6.88514891010830299e-01 -3.88964098337549247e-01 7.67894908294199964e-03
                    -7.85209296345330099e-01 -1.11183005024141593e-01 7.65837547845699997e-03
                    -4.87844461262681162e-01 -6.26106709232768455e-01 7.86487166401299981e-03
                    -2.93621303683378798e-01 -7.37702220466245118e-01 7.63034736577799990e-03
                    -5.80578341798693387e-01 -5.62043167569809876e-01 7.53216769862400000e-03
                    -7.77994878477669460e-01 -2.19195341205039390e-01 7.16274450921400036e-03
                    -1.95528110087736240e-01 -7.84419959952222912e-01 7.23749359942999979e-03
                    -6.67562406662079555e-01 -4.94058867584967087e-01 6.88873705365400033e-03
                    -9.70065467489013011e-02 -8.24824484789229539e-01 6.73787423472200012e-03
                    -7.67029080112698991e-01 -3.22253572954259937e-01 6.58760147645800001e-03
                    -8.52134631062725467e-01 -5.53572856340551131e-02 6.56397386974100043e-03
                    -4.59034776845581138e-01 -7.20502236320693323e-01 6.75929695408600002e-03
                    -3.60476062053627533e-01 -7.75748502109505633e-01 6.62734020038700019e-03
                    -7.46614051161292425e-01 -4.25129492237797668e-01 5.93418824640699980e-03
                    -5.54058683467657298e-01 -6.58521549423318753e-01 6.56207460755499957e-03
                    -8.45686112457160410e-01 -1.63577885739557644e-01 6.11001174754800033e-03
                    -2.61474565558392325e-01 -8.23410980567718997e-01 6.14286717416199970e-03
                    -6.43295885266826351e-01 -5.90899505408984838e-01 6.07294624840599984e-03
                    -8.35299944248545145e-01 -2.64651765964597996e-01 5.42167623554899985e-03
                    -1.65694454853941547e-01 -8.62333888244206537e-01 5.37501544402300014e-03
                    -8.14861365182181330e-01 -3.62422812612702039e-01 4.59568821999400013e-03
                    -7.24916525851842852e-01 -5.19796951937957386e-01 5.33618992961100002e-03
                    -7.57696286253323886e-02 -8.92607674214299651e-01 4.56993737540900037e-03
                    -4.42242634627807663e-01 -7.92761734691978970e-01 5.44663241367300023e-03
                    -9.02682415047762921e-01 -1.12551640683922982e-01 5.27442327957799995e-03
                    -5.38503350869325104e-01 -7.33410787171000722e-01 5.35090579030399974e-03
                    -3.41883918511743823e-01 -8.44245053680024005e-01 5.27660095420900022e-03
                    -7.97263860238179189e-01 -4.48734093482190399e-01 4.30018157389400017e-03
                    -6.28891758057308015e-01 -6.66846417008255976e-01 4.99824430716199961e-03
                    -8.91410221869499919e-01 -2.20462665262784352e-01 4.69853362942899977e-03
                    -2.39303273920731696e-01 -8.87478489699914275e-01 4.85981655752300030e-03
                    -8.67733073743958361e-01 -3.26426035545693605e-01 3.92960116374199957e-03
                    -7.11727954675998697e-01 -5.94936671603205958e-01 4.42335154098000023e-03
                    -1.37622048239333966e-01 -9.21395318784136386e-01 4.15821285139300004e-03
                    -7.85705451047371217e-01 -5.19485604669765788e-01 3.62831066641600012e-03
                    -8.54923285223178286e-01 -4.09412039322762766e-01 3.54980989056999981e-03
                    -9.47934243590281445e-01 -5.66124354624876550e-02 4.04533690231499970e-03
                    -5.10701619053918407e-02 -9.48461845643240653e-01 2.74997177325300018e-03
                    -4.31400732085064520e-01 -8.46502493858650396e-01 3.96259596692300018e-03
                    -5.27044895246795764e-01 -7.90846040296282693e-01 3.95027933900699967e-03
                    -3.31190242097535836e-01 -8.93371118167587630e-01 3.78389000269900013e-03
                    -9.38336052632894435e-01 -1.67229564939150727e-01 3.73662685985800018e-03
                    -6.17099306322382080e-01 -7.26599206167858225e-01 3.74011918994900019e-03
                    -9.14313291474611156e-01 -2.79786343589928654e-01 3.24965209930600006e-03
                    -2.27512191539794933e-01 -9.31494825820701200e-01 3.42690501398499985e-03
                    -7.00088454217312028e-01 -6.55272608849804872e-01 3.33863712365500007e-03
                    -7.74610359381933677e-01 -5.78055097179580812e-01 2.75742228760000006e-03
                    -1.24207053402499726e-01 -9.59457065285587496e-01 2.61632989023800021e-03
                    -8.43237196597550698e-01 -4.78498653963401688e-01 2.71174567446600019e-03
                    -9.03729154890727804e-01 -3.60847382789604432e-01 2.63376060625700008e-03
                    -9.52071689063728854e-01 -2.30081967668126497e-01 2.36415247803799982e-03
                    -5.25226236432522087e-01 -8.26898477180594504e-01 2.47171238457499992e-03
                    -4.30757069856774422e-01 -8.79854730654636819e-01 2.51474341781900016e-03
                    -9.73483764641486449e-01 -1.13088218146317698e-01 2.56557226477100017e-03
                    -6.12063267272664691e-01 -7.66053023865866822e-01 2.31192320871599979e-03
                    -3.29630732449447572e-01 -9.23991279910993946e-01 2.39814690487599996e-03
                    -6.91255236876849422e-01 -6.98473608830743165e-01 2.06148693511099988e-03
                    -2.22977164564220259e-01 -9.58713847167306121e-01 2.10424687186700019e-03
                    -7.62788131249490409e-01 -6.24155041982559555e-01 1.72521161714499997e-03
                    -1.05994309894318461e-01 -9.80967222184524035e-01 1.70640551365699999e-03
                    -8.28829959863797483e-01 -5.35834627117448115e-01 1.69321452361100000e-03
                    -8.88701635210429775e-01 -4.31501747167771732e-01 1.68331112853099991e-03
                    -9.39853926911746629e-01 -3.12587613527860353e-01 1.54299866912100000e-03
                    -9.77851086434011862e-01 -1.82993407168899808e-01 1.18604652062799997e-03
                    -9.94313150889492681e-01 -5.61190736687032996e-02 1.15476896378499997e-03
                    -5.37787545783237619e-01 -8.38476863760304170e-01 1.12924676460900001e-03
                    -4.39228604881380880e-01 -8.94056747347730552e-01 1.12198085237100002e-03
                    -6.30028132364980342e-01 -7.71845513335237543e-01 1.07453738060199998e-03
                    -3.36063190586354199e-01 -9.38012850425386291e-01 1.02961909163399993e-03
                    -7.15075772267507848e-01 -6.94574875831306726e-01 9.34856278170000023e-04
                    -1.31524020318368795e-01 -9.88270222271415144e-01 6.98243074158999989e-04
                    -2.31420100207260337e-01 -9.69905039743745201e-01 7.97390131281999954e-04
                    -7.92689859343882253e-01 -6.05302780969197096e-01 8.10401076300000045e-04
                    -8.60358204121903003e-01 -5.04872205483958947e-01 7.63219634713999985e-04
                    -9.15903427116124380e-01 -3.95900950922521588e-01 6.92280023180999959e-04
                    -9.59291700177966389e-01 -2.78024641686340890e-01 4.58817188194999984e-04
                    -9.89547932751443038e-01 -1.38330978668465776e-01 1.75322281775999993e-04
                    -5.27738387936428008e-02 -9.97940203891528710e-01 2.48540944503000010e-04
                    ];

        end

    case 65

        xyw=[
            0.00000000000000000e+00 -0.00000000000000000e+00 1.00226137880520001e-02
            1.07373184221838003e-01 -0.00000000000000000e+00 9.94626755579999973e-03
            1.60447931815638989e-01 9.26290383902479975e-02 9.79444233160300069e-03
            2.13515295301397179e-01 -8.97355323000851512e-06 9.71896770741999955e-03
            2.65366092491079952e-01 9.19134147288371006e-02 9.49419737937499952e-03
            2.12308327489296789e-01 1.83833599351387850e-01 9.49428188197500023e-03
            3.17197010218074560e-01 -2.72284880961420130e-05 9.34589790786599997e-03
            3.15405132742755223e-01 1.82074476293511311e-01 9.12531594206300069e-03
            3.67209160525535649e-01 9.08629140537215441e-02 9.05219022291799992e-03
            2.62350516526173194e-01 2.72554752903606257e-01 9.05280088815500067e-03
            4.17192161854162968e-01 -3.67676309102168199e-05 8.83561466822400075e-03
            4.14828436741208229e-01 1.79673905310374665e-01 8.61991709752400054e-03
            3.63052058100112762e-01 2.69416141362850137e-01 8.62113246893100003e-03
            4.64750234574498711e-01 8.95260136369763759e-02 8.47821467384500017e-03
            3.09965869910308056e-01 3.57737922590889990e-01 8.48050352790899985e-03
            5.12279044869404898e-01 -8.70162475921704496e-07 8.20004925499000052e-03
            4.59517716027207479e-01 2.65329723205908530e-01 8.05948340522799969e-03
            5.09361528023961729e-01 1.76705569569325649e-01 7.98860777127399953e-03
            4.07740642266395448e-01 3.52861999627508083e-01 7.99399700476799945e-03
            5.56772531032016960e-01 8.79789475760601875e-02 7.78510859607499991e-03
            3.54549681070316147e-01 4.38322827281723448e-01 7.79153804155899993e-03
            6.01241598976819280e-01 1.27120228064913786e-04 7.45480038135800001e-03
            5.50561999126976120e-01 2.60367612243769975e-01 7.37816930418999997e-03
            5.00789481221875610e-01 3.46685079626586790e-01 7.38469744659799986e-03
            5.97804583822010183e-01 1.73275604143368095e-01 7.24375472062699973e-03
            4.48946575316769236e-01 4.31283551620397942e-01 7.26025998860699989e-03
            6.42068219761605108e-01 8.63355181759781209e-02 6.98810701843300026e-03
            3.95528445260798667e-01 5.13223844610933622e-01 7.00503164311299959e-03
            5.88068765413905070e-01 3.39089445067025319e-01 6.66147346365399996e-03
            6.82867075753078256e-01 4.16399799497144400e-04 6.62018814704300011e-03
            6.35093108670507389e-01 2.54503078696227070e-01 6.58726499455100038e-03
            5.38342180885458155e-01 4.22328521934154821e-01 6.61340786323900021e-03
            6.79020191356738079e-01 1.69442120902228988e-01 6.39638591147699987e-03
            4.86338747165921514e-01 5.03324888338738230e-01 6.44249850973500036e-03
            4.32461708551271307e-01 5.81192659040216331e-01 6.14737976983999960e-03
            7.19476623516990488e-01 8.47360571352952091e-02 6.10376872736300043e-03
            6.22267341151800712e-01 4.10412730243512325e-01 5.86173344265100010e-03
            6.69030792671009400e-01 3.29115665883846387e-01 5.83808778909299976e-03
            5.72586341186540060e-01 4.89916791811229047e-01 5.77278755611299998e-03
            7.12467408068516939e-01 2.46865827804276489e-01 5.69636220519499983e-03
            7.55916640347585878e-01 8.50121476065331593e-04 5.72252496195499970e-03
            5.20283026377830971e-01 5.66796614655045383e-01 5.57219102781500008e-03
            7.52278319325379474e-01 1.64552448176672667e-01 5.45422707722499969e-03
            4.65531892096713362e-01 6.40248614908727309e-01 5.24677651235799965e-03
            7.88159761462871589e-01 8.26946384219095776e-02 5.15340411745500004e-03
            6.99881378240518881e-01 3.95037023514096919e-01 5.05634317520200008e-03
            6.53273763467947077e-01 4.72712219704802428e-01 5.01704054771999995e-03
            7.43092088422989239e-01 3.15650253531821801e-01 4.97901306948099996e-03
            6.03625070398092101e-01 5.48055612410321524e-01 4.89893369505600002e-03
            7.82384731457434701e-01 2.35703238067039633e-01 4.76094733620200002e-03
            8.19101250236940026e-01 -7.42367946815627002e-05 4.78188052084799978e-03
            5.50660977391418505e-01 6.20533472448488999e-01 4.67690542486600032e-03
            8.17459684903545258e-01 1.56679456600684663e-01 4.46239325266899970e-03
            4.94994424761754315e-01 6.89268124582391284e-01 4.31356067553300034e-03
            8.47466190616510184e-01 7.71668590329569887e-02 4.15453096751899984e-03
            7.18810667545411786e-01 4.65642450063486590e-01 4.25109942447000032e-03
            7.63267141389213810e-01 3.88622534408707754e-01 4.35357824810699966e-03
            6.72241010643340742e-01 5.38309325151412388e-01 4.05420774068400002e-03
            8.04147837004753008e-01 3.08660622287153796e-01 4.24883965064200026e-03
            6.23171538966466376e-01 6.06799487703870266e-01 3.85928354023799998e-03
            8.40571838826404405e-01 2.28623751191902314e-01 3.97971797841099959e-03
            8.71946709477104287e-01 -4.73817824490899240e-03 3.78841810271199992e-03
            5.70566649613250765e-01 6.72162278956330628e-01 3.64448284283200004e-03
            8.71522451152963229e-01 1.49118656480054207e-01 3.64139337464400000e-03
            5.15931551256481602e-01 7.34035310271856423e-01 3.28751680285900008e-03
            8.95633473898481580e-01 6.83437587850267980e-02 3.22961125591499998e-03
            7.88732127999880128e-01 4.35172399673382060e-01 2.57245090472699991e-03
            7.57984430279080246e-01 4.87000221592990390e-01 2.73445598490699993e-03
            8.23630655542334167e-01 3.68943815694714128e-01 3.38666652033900017e-03
            7.15568525306702785e-01 5.52740226175492011e-01 3.43631424158299991e-03
            8.60320416156517553e-01 2.90484776852564963e-01 3.43065871148700002e-03
            6.63403277810093228e-01 6.23610961425395516e-01 3.39408159544900018e-03
            9.16333192172601119e-01 -5.52736933171844771e-03 2.83115555419499992e-03
            8.92670655662287427e-01 2.09141409269853806e-01 3.19234771544299988e-03
            6.06853945474310374e-01 6.90801722017678199e-01 3.06769483519199991e-03
            9.18041931253040677e-01 1.25181842976169605e-01 2.86132808750100016e-03
            5.49938292151845087e-01 7.51107408702582635e-01 2.52106724557500010e-03
            8.30114749198946344e-01 4.36119213979587028e-01 3.04549503190999988e-03
            7.82879130870570816e-01 5.16722743112057459e-01 3.05234104169899980e-03
            9.38286081731217303e-01 4.79846902241694157e-02 2.28570211332600019e-03
            8.71273070030332386e-01 3.53204100029559043e-01 2.92610580844900004e-03
            7.30258869805446853e-01 5.93682526257915288e-01 2.91222693355400013e-03
            9.06217832643674170e-01 2.69061631030399762e-01 2.69843874117499990e-03
            6.72790039537225937e-01 6.65277125044756779e-01 2.65761800614299978e-03
            5.06840979606803188e-01 7.99318070605961384e-01 1.99917921561299991e-03
            9.33944956946977878e-01 1.82302211267787673e-01 2.44583196925200015e-03
            6.11828161933012926e-01 7.30312438318991841e-01 2.26589034185999995e-03
            9.54610593834458676e-01 9.70477674411851787e-02 1.99056338398399986e-03
            5.61432442614396954e-01 7.84357649663082412e-01 1.76934970702599995e-03
            9.65148730235607988e-01 5.83073390153666851e-03 1.76460618128099992e-03
            8.31349873283759422e-01 4.92175994236409398e-01 2.24662934487400000e-03
            8.74854825236558442e-01 4.11264420209463355e-01 2.19547006801600010e-03
            7.81401909986196519e-01 5.69944682364613264e-01 2.20066282864300007e-03
            9.11659253994095753e-01 3.28213312842733973e-01 2.05677144082100014e-03
            7.25130156832938910e-01 6.43677482131609580e-01 2.06229745077100022e-03
            9.41613731654346253e-01 2.42018995885192545e-01 1.88319400533199996e-03
            6.62778534581157763e-01 7.12019599624631905e-01 1.82986657465800009e-03
            9.64081881579468281e-01 1.53750893290202972e-01 1.62690199042700000e-03
            6.05033796931687351e-01 7.70633794722112686e-01 1.33861458188700008e-03
            9.77946153367900473e-01 6.03808937671692220e-02 1.42019616970100009e-03
            5.26651543585005988e-01 8.27874301313570538e-01 1.39036118836199994e-03
            8.64881229285689113e-01 4.73755671175498660e-01 1.42178717363199993e-03
            8.17679250252735357e-01 5.51658930292705518e-01 1.41011628136399991e-03
            9.05253162176683945e-01 3.92522295174929481e-01 1.37341746888800008e-03
            7.63543916101214193e-01 6.25770048864424222e-01 1.34371539877700002e-03
            9.38798949431799867e-01 3.07286607845038751e-01 1.28191954890700000e-03
            7.01761249508369600e-01 6.96043134758961957e-01 1.24608003992800003e-03
            9.65095662701151880e-01 2.18042040249806862e-01 1.14558934658899993e-03
            9.83209118931755888e-01 1.23791243071380105e-01 1.00978328234600009e-03
            6.36793081128479543e-01 7.59500496713096274e-01 9.20176790160000019e-04
            9.91674495391106414e-01 2.26365158155490756e-02 9.29646549994999965e-04
            5.62875847282320652e-01 8.17384200397119898e-01 8.56589133727000008e-04
            8.52040712508415066e-01 5.18415014241330518e-01 6.15386714708999983e-04
            8.95628071786307878e-01 4.38934044180895655e-01 6.03871648948999971e-04
            8.01054327269182020e-01 5.94386253512860452e-01 5.94845518337000045e-04
            9.32000590079671509e-01 3.55687518702678895e-01 5.70125663872000010e-04
            7.42479936307709143e-01 6.66372334417594137e-01 5.59214403358000039e-04
            9.61058159177462845e-01 2.68430041119814888e-01 5.16461052757000009e-04
            6.77366201148239533e-01 7.33007040887932559e-01 4.69050144477000015e-04
            9.82484501226762119e-01 1.76200902451200042e-01 4.57922790376999998e-04
            9.95289510146307288e-01 7.82347486599258846e-02 4.21940731861999974e-04
            5.19303001975390144e-01 8.52871980961562470e-01 3.86169486359000012e-04
            6.03593072093222216e-01 7.95455299874740285e-01 3.78486748618999996e-04
            5.36865921109190153e-02 9.29879052213381657e-02 9.94626755579999973e-03
            4.86553374071624115e-06 1.85266504132140825e-01 9.79444233160300069e-03
            1.06765418975758003e-01 1.84905183050931121e-01 9.71896770741999955e-03
            5.30836941417922731e-02 2.75770484764704726e-01 9.49419737937499952e-03
            -5.30504033627839724e-02 2.75781204716410988e-01 9.49428188197500023e-03
            1.58622085671435237e-01 2.74687054609276615e-01 9.34589790786599997e-03
            2.14445204493233542e-05 3.64186095585984693e-01 9.12531594206300069e-03
            1.04914988430362899e-01 3.63443918544332500e-01 9.05219022291799992e-03
            -1.04864081673626874e-01 3.63479588459438341e-01 9.05280088815500067e-03
            2.08627922629486762e-01 3.61280626609999211e-01 8.83561466822400075e-03
            5.18120519746599409e-02 4.49088917085259665e-01 8.61991709752400054e-03
            -5.17951935597512625e-02 4.49120375892346713e-01 8.62113246893100003e-03
            1.54843315178075758e-01 4.47248516374780969e-01 8.47821467384500017e-03
            -1.54827193905627658e-01 4.47307278943914366e-01 8.48050352790899985e-03
            2.56140276017512158e-01 4.43646231602094987e-01 8.20004925499000052e-03
            -2.34226618063990914e-05 5.30618877171519676e-01 8.05948340522799969e-03
            1.01649251774746485e-01 5.29472807763872955e-01 7.98860777127399953e-03
            -1.01717134574399265e-01 5.29544754171835552e-01 7.99399700476799945e-03
            2.02194261916921053e-01 5.26168629791116449e-01 7.78510859607499991e-03
            -2.02323862949433220e-01 5.26210444351426165e-01 7.79153804155899993e-03
            3.00510710141570547e-01 5.20754058639933848e-01 7.45480038135800001e-03
            4.97960330376871693e-02 6.06984483724192225e-01 7.37816930418999997e-03
            -4.98433454587172212e-02 6.07038952499467821e-01 7.38469744659799986e-03
            1.48841216866752224e-01 6.04351758160328711e-01 7.24375472062699973e-03
            -1.49029224279257161e-01 6.04440914976544952e-01 7.26025998860699989e-03
            2.46265357891512437e-01 5.99215148364188877e-01 6.98810701843300026e-03
            -2.46700664630586375e-01 5.99149603820681276e-01 7.00503164311299959e-03
            3.74309123740901935e-04 6.78827212554106163e-01 6.66147346365399996e-03
            3.41072925072043953e-01 5.91588434909907090e-01 6.62018814704300011e-03
            9.71404228429708777e-02 6.77258305225204138e-01 6.58726499455100038e-03
            -9.65761382949824471e-02 6.77382265542601614e-01 6.61340786323900021e-03
            1.92768914505924538e-01 6.72769795848620511e-01 6.39638591147699987e-03
            -1.92722766075352409e-01 6.72844154059754396e-01 6.44249850973500036e-03
            -2.87096752946219169e-01 6.65119155289531139e-01 6.14737976983999960e-03
            2.86354733662800065e-01 6.65453061962413717e-01 6.10376872736300043e-03
            -4.42941798515111493e-02 7.44105690504613504e-01 5.86173344265100010e-03
            4.94928688966622884e-02 7.43955495309057246e-01 5.83808778909299976e-03
            -1.37987216855826350e-01 7.40832713233142148e-01 5.77278755611299998e-03
            1.42441625829480362e-01 7.40447788657928152e-01 5.69636220519499983e-03
            3.77222093379217749e-01 6.55068074422426982e-01 5.72252496195499970e-03
            -2.30718753881372945e-01 7.33976625328573662e-01 5.57219102781500008e-03
            2.33632559286768987e-01 7.33768359340376874e-01 5.45422707722499969e-03
            -3.21705619200401360e-01 7.23286752281953604e-01 5.24677651235799965e-03
            3.22464223101293523e-01 7.23913694878484892e-01 5.15340411745500004e-03
            7.82859132166083965e-03 8.03633564949003287e-01 5.05634317520200008e-03
            -8.27439092097162210e-02 8.02107784641510002e-01 5.01704054771999995e-03
            9.81849059419381692e-02 8.01361752691451823e-01 4.97901306948099996e-03
            -1.72817547834930363e-01 7.96782451531078628e-01 4.89893369505600002e-03
            1.87067373808409904e-01 7.95416672008724346e-01 4.76094733620200002e-03
            4.09614916068559931e-01 7.09325372579443614e-01 4.78188052084799978e-03
            -2.62067262343253060e-01 7.87153131517981386e-01 4.67690542486600032e-03
            2.73041452784438365e-01 7.86280581996435024e-01 4.46239325266899970e-03
            -3.49426493526330872e-01 7.73311808866539940e-01 4.31356067553300034e-03
            3.56904635055461827e-01 7.72510679438801695e-01 4.15453096751899984e-03
            -4.38528570627001874e-02 8.55329523637320510e-01 4.25109942447000032e-03
            4.50765834135739910e-02 8.55320001521342022e-01 4.35357824810699966e-03
            -1.30069045353510104e-01 8.51332455258564491e-01 4.05420774068400002e-03
            1.34765978453788238e-01 8.50742766388001059e-01 4.24883965064200026e-03
            -2.13918001871701485e-01 8.43082127512339352e-01 3.85928354023799998e-03
            2.22291942972522000e-01 8.42268441725416150e-01 3.97971797841099959e-03
            4.40076737466302159e-01 7.52758912030967342e-01 3.78841810271199992e-03
            -2.96826284235199356e-01 8.30206352595415087e-01 3.64448284283200004e-03
            3.06620680886549757e-01 8.29319910906975766e-01 3.64139337464400000e-03
            -3.77727450341979076e-01 8.13827485137954443e-01 3.28751680285900008e-03
            3.88629305651291723e-01 8.09813220268305378e-01 3.22961125591499998e-03
            1.74957108569563403e-02 9.00648259465546719e-01 2.57245090472699991e-03
            -4.27623484086403632e-02 8.99933883091253439e-01 2.73445598490699993e-03
            9.23006108103808787e-02 8.97756978882648937e-01 3.38666652033900017e-03
            -1.20902814908181055e-01 8.96070634151918677e-01 3.43631424158299991e-03
            1.78593011911283850e-01 8.90301724212226975e-01 3.43065871148700002e-03
            -2.08361295767783367e-01 8.86329572250103892e-01 3.39408159544900018e-03
            4.62953438343667978e-01 7.90804138086501296e-01 2.83115555419499992e-03
            2.65213554420172193e-01 8.77646169651378871e-01 3.19234771544299988e-03
            -2.94824867508190036e-01 8.70951794176408445e-01 3.06769483519199991e-03
            3.50610309516602880e-01 8.57638555692545212e-01 2.86132808750100016e-03
            -3.75508950831214716e-01 8.51814235868617708e-01 2.52106724557500010e-03
            3.73670562146493843e-02 9.36960067852229073e-01 3.04549503190999988e-03
            -5.60554568129369366e-02 9.36354586982625192e-01 3.05234104169899980e-03
            4.27587080138751130e-01 8.36571927908680846e-01 2.28570211332600019e-03
            1.29752811668748164e-01 9.31146662294305538e-01 2.92610580844900004e-03
            -1.49014714619553124e-01 9.29263995719387559e-01 2.91222693355400013e-03
            2.20094708665835637e-01 9.19338479947096610e-01 2.69843874117499990e-03
            -2.39751871076822859e-01 9.15291828174752964e-01 2.65761800614299978e-03
            -4.38809265045324748e-01 8.38596199321462787e-01 1.99917921561299991e-03
            3.09094132349507167e-01 8.99971164086340480e-01 2.44583196925200015e-03
            -3.26555043317496241e-01 8.95014950144224386e-01 2.26589034185999995e-03
            3.93259464932598835e-01 8.75240908702982434e-01 1.99056338398399986e-03
            -3.98557428953685677e-01 8.78393582644357962e-01 1.76934970702599995e-03
            4.77524801436366164e-01 8.38758685765098932e-01 1.76460618128099992e-03
            -1.05619774997141400e-02 9.66058106814914330e-01 2.24662934487400000e-03
            8.12619770442057354e-02 9.63278713382986651e-01 2.19547006801600010e-03
            -1.02885618686509556e-01 9.61686245796034012e-01 2.20066282864300007e-03
            1.71588560214991054e-01 9.53626729975423970e-01 2.05677144082100014e-03
            -1.94875972953508392e-01 9.49819877933324008e-01 2.06229745077100022e-03
            2.61212267192194980e-01 9.36470910107523591e-01 1.88319400533199996e-03
            -2.85237793976777010e-01 9.29992847842621706e-01 1.82986657465800009e-03
            3.48888761345867993e-01 9.11794847421221966e-01 1.62690199042700000e-03
            -3.64871544778308188e-01 9.09291535652053007e-01 1.33861458188700008e-03
            4.36681688778372323e-01 8.77116659233459162e-01 1.42019616970100009e-03
            -4.53634404285341852e-01 8.70030766343688144e-01 1.39036118836199994e-03
            2.21561682179155783e-02 9.85886951405470002e-01 1.42178717363199993e-03
            -6.89110227316639679e-02 9.83960468012635037e-01 1.41011628136399991e-03
            1.12692301915079215e-01 9.80233382888667348e-01 1.37341746888800008e-03
            -1.60160801193413693e-01 9.74133452680917666e-01 1.34371539877700002e-03
            2.03281466079349965e-01 9.66667043176600682e-01 1.28191954890700000e-03
            -2.51910412076831569e-01 9.55764636845239046e-01 1.24608003992800003e-03
            2.93717885401254264e-01 9.44818381106278915e-01 1.14558934658899993e-03
            3.84398198200008478e-01 9.13379695763106070e-01 1.00978328234600009e-03
            -3.39350183776201075e-01 9.31229233567976489e-01 9.20176790160000019e-04
            4.76233449946119658e-01 8.70133563201586746e-01 9.29646549994999965e-04
            -4.26437558554775997e-01 8.96156883121739800e-01 8.56589133727000008e-04
            -2.29402157820560379e-02 9.97096409211546186e-01 6.15386714708999983e-04
            6.76860030466572560e-02 9.95103684599863203e-01 6.03871648948999971e-04
            -1.14226431567803433e-01 9.90926523982995522e-01 5.94845518337000045e-04
            1.57965868034263185e-01 9.84979946702421949e-01 5.70125663872000010e-04
            -2.05855401830921214e-01 9.76192653851525205e-01 5.59214403358000039e-04
            2.48061844840070406e-01 9.66515800921898927e-01 5.16461052757000009e-04
            -2.96119617987688488e-01 9.53119858303301704e-01 4.69050144477000015e-04
            3.38647792920898305e-01 9.38956988112459423e-01 4.57922790376999998e-04
            4.29891475274967405e-01 9.01063374236834691e-01 4.21940731861999974e-04
            -4.78957300700976030e-01 8.76165582452989589e-01 3.86169486359000012e-04
            -3.87087961219882537e-01 9.20454583918392855e-01 3.78486748618999996e-04
            -5.36865921109189806e-02 9.29879052213381796e-02 9.94626755579999973e-03
            -1.60443066281898278e-01 9.26374657418928132e-02 9.79444233160300069e-03
            -1.06749876325639162e-01 1.84914156604161162e-01 9.71896770741999955e-03
            -2.12282398349287665e-01 1.83857070035867709e-01 9.49419737937499952e-03
            -2.65358730852080726e-01 9.19476053650232494e-02 9.49428188197500023e-03
            -1.58574924546639351e-01 2.74714283097372780e-01 9.34589790786599997e-03
            -3.15383688222305891e-01 1.82111619292473464e-01 9.12531594206300069e-03
            -2.62294172095172640e-01 2.72581004490611067e-01 9.05219022291799992e-03
            -3.67214598199800069e-01 9.09248355558321397e-02 9.05280088815500067e-03
            -2.08564239224676262e-01 3.61317394240909473e-01 8.83561466822400075e-03
            -3.63016384766548239e-01 2.69415011774885138e-01 8.61991709752400054e-03
            -4.14847251659864080e-01 1.79704234529496687e-01 8.62113246893100003e-03
            -3.09906919396422897e-01 3.57722502737804704e-01 8.47821467384500017e-03
            -4.64793063815935770e-01 8.95693563530243625e-02 8.48050352790899985e-03
            -2.56138768851892795e-01 4.43647101764571006e-01 8.20004925499000052e-03
            -4.59541138689013973e-01 2.65289153965611146e-01 8.05948340522799969e-03
            -4.07712276249215300e-01 3.52767238194547306e-01 7.98860777127399953e-03
            -5.09457776840794851e-01 1.76682754544327497e-01 7.99399700476799945e-03
            -3.54578269115095823e-01 4.38189682215056442e-01 7.78510859607499991e-03
            -5.56873544019749422e-01 8.78876170697029668e-02 7.79153804155899993e-03
            -3.00730888835248567e-01 5.20626938411869022e-01 7.45480038135800001e-03
            -5.00765966089289027e-01 3.46616871480422528e-01 7.37816930418999997e-03
            -5.50632826680592879e-01 2.60353872872880976e-01 7.38469744659799986e-03
            -4.48963366955257903e-01 4.31076154016960700e-01 7.24375472062699973e-03
            -5.97975799596026536e-01 1.73157363356147065e-01 7.26025998860699989e-03
            -3.95802861870092615e-01 5.12879630188210922e-01 6.98810701843300026e-03
            -6.42229109891385042e-01 8.59257592097478901e-02 7.00503164311299959e-03
            -5.87694456290164258e-01 3.39737767487080955e-01 6.66147346365399996e-03
            -3.41794150681034359e-01 5.91172035110410055e-01 6.62018814704300011e-03
            -5.37952685827536414e-01 4.22755226528977235e-01 6.58726499455100038e-03
            -6.34918319180440505e-01 2.55053743608447070e-01 6.61340786323900021e-03
            -4.86251276850813485e-01 5.03327674946391634e-01 6.39638591147699987e-03
            -6.79061513241274062e-01 1.69519265721016110e-01 6.44249850973500036e-03
            -7.19558461497490476e-01 8.39264962493147804e-02 6.14737976983999960e-03
            -4.33121889854190478e-01 5.80717004827118632e-01 6.10376872736300043e-03
            -6.66561521003312007e-01 3.33692960261101179e-01 5.86173344265100010e-03
            -6.19537923774347021e-01 4.14839829425211082e-01 5.83808778909299976e-03
            -7.10573558042366327e-01 2.50915921421913268e-01 5.77278755611299998e-03
            -5.70025782239036660e-01 4.93581960853651802e-01 5.69636220519499983e-03
            -3.78694546968368129e-01 6.54217952946361869e-01 5.72252496195499970e-03
            -7.51001780259204166e-01 1.67180010673528251e-01 5.57219102781500008e-03
            -5.18645760038610515e-01 5.69215911163704402e-01 5.45422707722499969e-03
            -7.87237511297114834e-01 8.30381373732264616e-02 5.24677651235799965e-03
            -4.65695538361578121e-01 6.41219056456575509e-01 5.15340411745500004e-03
            -6.92052786918857854e-01 4.08596541434906479e-01 5.05634317520200008e-03
            -7.36017672677663159e-01 3.29395564936707796e-01 5.01704054771999995e-03
            -6.44907182481050834e-01 4.85711499159630355e-01 4.97901306948099996e-03
            -7.76442618233022519e-01 2.48726839120757354e-01 4.89893369505600002e-03
            -5.95317357649024936e-01 5.59713433941684824e-01 4.76094733620200002e-03
            -4.09486334168379984e-01 7.09399609374125384e-01 4.78188052084799978e-03
            -8.12728239734671787e-01 1.66619659069492471e-01 4.67690542486600032e-03
            -5.44418232119106782e-01 6.29601125395750638e-01 4.46239325266899970e-03
            -8.44420918288085298e-01 8.40436842841486703e-02 4.31356067553300034e-03
            -4.90561555561048357e-01 6.95343820405844970e-01 4.15453096751899984e-03
            -7.62663524608112153e-01 3.89687073573833975e-01 4.25109942447000032e-03
            -7.18190557975639798e-01 4.66697467112634434e-01 4.35357824810699966e-03
            -8.02310055996850902e-01 3.13023130107152103e-01 4.05420774068400002e-03
            -6.69381858550964881e-01 5.42082144100847318e-01 4.24883965064200026e-03
            -8.37089540838167889e-01 2.36282639808469336e-01 3.85928354023799998e-03
            -6.18279895853882211e-01 6.13644690533514003e-01 3.97971797841099959e-03
            -4.31869972010802017e-01 7.57497090275876417e-01 3.78841810271199992e-03
            -8.67392933848450065e-01 1.58044073639084792e-01 3.64448284283200004e-03
            -5.64901770266413528e-01 6.80201254426921587e-01 3.64139337464400000e-03
            -8.93659001598460900e-01 7.97921748660982560e-02 3.28751680285900008e-03
            -5.07004168247189746e-01 7.41469461483278747e-01 3.22961125591499998e-03
            -7.71236417142923902e-01 4.65475859792164659e-01 2.57245090472699991e-03
            -8.00746778687720484e-01 4.12933661498263327e-01 2.73445598490699993e-03
            -7.31330044731953288e-01 5.28813163187934920e-01 3.38666652033900017e-03
            -8.36471340214883785e-01 3.43330407976426832e-01 3.43631424158299991e-03
            -6.81727404245233815e-01 5.99816947359662178e-01 3.43065871148700002e-03
            -8.71764573577876734e-01 2.62718610824708598e-01 3.39408159544900018e-03
            -4.53379753828933141e-01 7.96331507418219897e-01 2.83115555419499992e-03
            -6.27457101242115289e-01 6.68504760381525176e-01 3.19234771544299988e-03
            -9.01678812982500411e-01 1.80150072158730634e-01 3.06769483519199991e-03
            -5.67431621736437575e-01 7.32456712716375802e-01 2.86132808750100016e-03
            -9.25447242983060026e-01 1.00706827166035143e-01 2.52106724557500010e-03
            -7.92747692984296926e-01 5.00840853872642211e-01 3.04549503190999988e-03
            -8.38934587683507593e-01 4.19631843870568066e-01 3.05234104169899980e-03
            -5.10699001592465840e-01 7.88587237684511666e-01 2.28570211332600019e-03
            -7.41520258361584084e-01 5.77942562264746829e-01 2.92610580844900004e-03
            -8.79273584425000143e-01 3.35581469461472326e-01 2.91222693355400013e-03
            -6.86123123977838589e-01 6.50276848916696903e-01 2.69843874117499990e-03
            -9.12541910614048879e-01 2.50014703129996241e-01 2.65761800614299978e-03
            -9.45650244652127769e-01 3.92781287155015840e-02 1.99917921561299991e-03
            -6.24850824597470655e-01 7.17668952818552919e-01 2.44583196925200015e-03
            -9.38383205250509222e-01 1.64702511825232739e-01 2.26589034185999995e-03
            -5.61351128901860008e-01 7.78193141261797283e-01 1.99056338398399986e-03
            -9.59989871568082687e-01 9.40359329812758271e-02 1.76934970702599995e-03
            -4.87623928799241824e-01 8.32927951863562277e-01 1.76460618128099992e-03
            -8.41911850783473525e-01 4.73882112578505210e-01 2.24662934487400000e-03
            -7.93592848192352429e-01 5.52014293173523685e-01 2.19547006801600010e-03
            -8.84287528672705880e-01 3.91741563431421080e-01 2.20066282864300007e-03
            -7.40070693779104727e-01 6.25413417132690164e-01 2.05677144082100014e-03
            -9.20006129786447358e-01 3.06142395801714651e-01 2.06229745077100022e-03
            -6.80401464462151440e-01 6.94451914222331101e-01 1.88319400533199996e-03
            -9.48016328557934940e-01 2.17973248217989857e-01 1.82986657465800009e-03
            -6.15193120233600177e-01 7.58043954131019104e-01 1.62690199042700000e-03
            -9.69905341709995428e-01 1.38657740929940598e-01 1.33861458188700008e-03
            -5.41264464589528149e-01 8.16735765466290120e-01 1.42019616970100009e-03
            -9.80285947870348062e-01 4.21564650301179192e-02 1.39036118836199994e-03
            -8.42725061067773451e-01 5.12131280229971564e-01 1.42178717363199993e-03
            -8.86590272984399297e-01 4.32301537719929685e-01 1.41011628136399991e-03
            -7.92560860261604772e-01 5.87711087713738145e-01 1.37341746888800008e-03
            -9.23704717294627997e-01 3.48363403816493666e-01 1.34371539877700002e-03
            -7.35517483352450041e-01 6.59380435331562098e-01 1.28191954890700000e-03
            -9.53671661585201336e-01 2.59721502086277034e-01 1.24608003992800003e-03
            -6.71377777299897449e-01 7.26776340856472358e-01 1.14558934658899993e-03
            -5.98810920731747132e-01 7.89588452691726284e-01 1.00978328234600009e-03
            -9.76143264904680619e-01 1.71728736854880576e-01 9.20176790160000019e-04
            -5.15441045444986812e-01 8.47497047386037927e-01 9.29646549994999965e-04
            -9.89313405837096593e-01 7.87726827246200967e-02 8.56589133727000008e-04
            -8.74980928290471249e-01 4.78681394970215779e-01 6.15386714708999983e-04
            -8.27942068739650594e-01 5.56169640418967881e-01 6.03871648948999971e-04
            -9.15280758836985675e-01 3.96540270470135126e-01 5.94845518337000045e-04
            -7.74034722045408130e-01 6.29292427999743387e-01 5.70125663872000010e-04
            -9.48335338138630468e-01 3.09820319433931068e-01 5.59214403358000039e-04
            -7.12996314337392412e-01 6.98085759802084316e-01 5.16461052757000009e-04
            -9.73485819135928021e-01 2.20112817415369116e-01 4.69050144477000015e-04
            -6.43836708305864036e-01 7.62756085661259520e-01 4.57922790376999998e-04
            -5.65398034871339772e-01 8.22828625576909056e-01 4.21940731861999974e-04
            -9.98260302676366229e-01 2.32936014914276179e-02 3.86169486359000012e-04
            -9.90681033313104864e-01 1.24999284043652820e-01 3.78486748618999996e-04
            -1.07373184221838003e-01 1.31494226371532992e-17 9.94626755579999973e-03
            -1.60447931815638989e-01 -9.26290383902479419e-02 9.79444233160300069e-03
            -2.13515295301397179e-01 8.97355323001746318e-06 9.71896770741999955e-03
            -2.65366092491079952e-01 -9.19134147288370867e-02 9.49419737937499952e-03
            -2.12308327489296844e-01 -1.83833599351387766e-01 9.49428188197500023e-03
            -3.17197010218074560e-01 2.72284880961322823e-05 9.34589790786599997e-03
            -3.15405132742755223e-01 -1.82074476293511256e-01 9.12531594206300069e-03
            -3.67209160525535649e-01 -9.08629140537214469e-02 9.05219022291799992e-03
            -2.62350516526173250e-01 -2.72554752903606201e-01 9.05280088815500067e-03
            -4.17192161854162968e-01 3.67676309101800858e-05 8.83561466822400075e-03
            -4.14828436741208173e-01 -1.79673905310374721e-01 8.61991709752400054e-03
            -3.63052058100112818e-01 -2.69416141362850137e-01 8.62113246893100003e-03
            -4.64750234574498711e-01 -8.95260136369763204e-02 8.47821467384500017e-03
            -3.09965869910308223e-01 -3.57737922590889879e-01 8.48050352790899985e-03
            -5.12279044869404898e-01 8.70162476015735635e-07 8.20004925499000052e-03
            -4.59517716027207479e-01 -2.65329723205908530e-01 8.05948340522799969e-03
            -5.09361528023961729e-01 -1.76705569569325677e-01 7.98860777127399953e-03
            -4.07740642266395448e-01 -3.52861999627508027e-01 7.99399700476799945e-03
            -5.56772531032016960e-01 -8.79789475760600209e-02 7.78510859607499991e-03
            -3.54549681070316258e-01 -4.38322827281723337e-01 7.79153804155899993e-03
            -6.01241598976819280e-01 -1.27120228064767039e-04 7.45480038135800001e-03
            -5.50561999126976231e-01 -2.60367612243769864e-01 7.37816930418999997e-03
            -5.00789481221875721e-01 -3.46685079626586623e-01 7.38469744659799986e-03
            -5.97804583822010183e-01 -1.73275604143368012e-01 7.24375472062699973e-03
            -4.48946575316769236e-01 -4.31283551620397887e-01 7.26025998860699989e-03
            -6.42068219761605219e-01 -8.63355181759779544e-02 6.98810701843300026e-03
            -3.95528445260798667e-01 -5.13223844610933622e-01 7.00503164311299959e-03
            -5.88068765413905070e-01 -3.39089445067025264e-01 6.66147346365399996e-03
            -6.82867075753078256e-01 -4.16399799497070458e-04 6.62018814704300011e-03
            -6.35093108670507500e-01 -2.54503078696226959e-01 6.58726499455100038e-03
            -5.38342180885458266e-01 -4.22328521934154655e-01 6.61340786323900021e-03
            -6.79020191356738190e-01 -1.69442120902228821e-01 6.39638591147699987e-03
            -4.86338747165921736e-01 -5.03324888338738119e-01 6.44249850973500036e-03
            -4.32461708551271473e-01 -5.81192659040216109e-01 6.14737976983999960e-03
            -7.19476623516990488e-01 -8.47360571352952369e-02 6.10376872736300043e-03
            -6.22267341151800712e-01 -4.10412730243512325e-01 5.86173344265100010e-03
            -6.69030792671009511e-01 -3.29115665883846220e-01 5.83808778909299976e-03
            -5.72586341186540060e-01 -4.89916791811229047e-01 5.77278755611299998e-03
            -7.12467408068517050e-01 -2.46865827804276350e-01 5.69636220519499983e-03
            -7.55916640347585878e-01 -8.50121476065224474e-04 5.72252496195499970e-03
            -5.20283026377830971e-01 -5.66796614655045383e-01 5.57219102781500008e-03
            -7.52278319325379585e-01 -1.64552448176672528e-01 5.45422707722499969e-03
            -4.65531892096713418e-01 -6.40248614908727198e-01 5.24677651235799965e-03
            -7.88159761462871589e-01 -8.26946384219096192e-02 5.15340411745500004e-03
            -6.99881378240518992e-01 -3.95037023514096752e-01 5.05634317520200008e-03
            -6.53273763467947188e-01 -4.72712219704802261e-01 5.01704054771999995e-03
            -7.43092088422989239e-01 -3.15650253531821690e-01 4.97901306948099996e-03
            -6.03625070398092101e-01 -5.48055612410321413e-01 4.89893369505600002e-03
            -7.82384731457434701e-01 -2.35703238067039578e-01 4.76094733620200002e-03
            -8.19101250236940026e-01 7.42367946817279191e-05 4.78188052084799978e-03
            -5.50660977391418505e-01 -6.20533472448488999e-01 4.67690542486600032e-03
            -8.17459684903545258e-01 -1.56679456600684525e-01 4.46239325266899970e-03
            -4.94994424761754592e-01 -6.89268124582391062e-01 4.31356067553300034e-03
            -8.47466190616510184e-01 -7.71668590329568083e-02 4.15453096751899984e-03
            -7.18810667545411786e-01 -4.65642450063486590e-01 4.25109942447000032e-03
            -7.63267141389213810e-01 -3.88622534408707587e-01 4.35357824810699966e-03
            -6.72241010643340853e-01 -5.38309325151412166e-01 4.05420774068400002e-03
            -8.04147837004753008e-01 -3.08660622287153796e-01 4.24883965064200026e-03
            -6.23171538966466598e-01 -6.06799487703870044e-01 3.85928354023799998e-03
            -8.40571838826404405e-01 -2.28623751191902119e-01 3.97971797841099959e-03
            -8.71946709477104287e-01 4.73817824490908434e-03 3.78841810271199992e-03
            -5.70566649613250543e-01 -6.72162278956330739e-01 3.64448284283200004e-03
            -8.71522451152963340e-01 -1.49118656480053929e-01 3.64139337464400000e-03
            -5.15931551256481602e-01 -7.34035310271856312e-01 3.28751680285900008e-03
            -8.95633473898481580e-01 -6.83437587850267425e-02 3.22961125591499998e-03
            -7.88732127999880239e-01 -4.35172399673381782e-01 2.57245090472699991e-03
            -7.57984430279080246e-01 -4.87000221592990501e-01 2.73445598490699993e-03
            -8.23630655542334167e-01 -3.68943815694714017e-01 3.38666652033900017e-03
            -7.15568525306702896e-01 -5.52740226175491900e-01 3.43631424158299991e-03
            -8.60320416156517664e-01 -2.90484776852564797e-01 3.43065871148700002e-03
            -6.63403277810093339e-01 -6.23610961425395405e-01 3.39408159544900018e-03
            -9.16333192172601119e-01 5.52736933171862725e-03 2.83115555419499992e-03
            -8.92670655662287427e-01 -2.09141409269853862e-01 3.19234771544299988e-03
            -6.06853945474310263e-01 -6.90801722017678310e-01 3.06769483519199991e-03
            -9.18041931253040677e-01 -1.25181842976169355e-01 2.86132808750100016e-03
            -5.49938292151845198e-01 -7.51107408702582524e-01 2.52106724557500010e-03
            -8.30114749198946456e-01 -4.36119213979587028e-01 3.04549503190999988e-03
            -7.82879130870570927e-01 -5.16722743112057237e-01 3.05234104169899980e-03
            -9.38286081731217303e-01 -4.79846902241691797e-02 2.28570211332600019e-03
            -8.71273070030332497e-01 -3.53204100029558765e-01 2.92610580844900004e-03
            -7.30258869805447075e-01 -5.93682526257915066e-01 2.91222693355400013e-03
            -9.06217832643674170e-01 -2.69061631030399817e-01 2.69843874117499990e-03
            -6.72790039537225937e-01 -6.65277125044756779e-01 2.65761800614299978e-03
            -5.06840979606803521e-01 -7.99318070605961273e-01 1.99917921561299991e-03
            -9.33944956946977989e-01 -1.82302211267787534e-01 2.44583196925200015e-03
            -6.11828161933013037e-01 -7.30312438318991730e-01 2.26589034185999995e-03
            -9.54610593834458676e-01 -9.70477674411852204e-02 1.99056338398399986e-03
            -5.61432442614397065e-01 -7.84357649663082301e-01 1.76934970702599995e-03
            -9.65148730235607988e-01 -5.83073390153675437e-03 1.76460618128099992e-03
            -8.31349873283759533e-01 -4.92175994236409287e-01 2.24662934487400000e-03
            -8.74854825236558442e-01 -4.11264420209463077e-01 2.19547006801600010e-03
            -7.81401909986196630e-01 -5.69944682364613042e-01 2.20066282864300007e-03
            -9.11659253994095864e-01 -3.28213312842733806e-01 2.05677144082100014e-03
            -7.25130156832938910e-01 -6.43677482131609469e-01 2.06229745077100022e-03
            -9.41613731654346364e-01 -2.42018995885192240e-01 1.88319400533199996e-03
            -6.62778534581157763e-01 -7.12019599624631794e-01 1.82986657465800009e-03
            -9.64081881579468281e-01 -1.53750893290202945e-01 1.62690199042700000e-03
            -6.05033796931687129e-01 -7.70633794722112797e-01 1.33861458188700008e-03
            -9.77946153367900473e-01 -6.03808937671690901e-02 1.42019616970100009e-03
            -5.26651543585005877e-01 -8.27874301313570649e-01 1.39036118836199994e-03
            -8.64881229285689224e-01 -4.73755671175498494e-01 1.42178717363199993e-03
            -8.17679250252735468e-01 -5.51658930292705407e-01 1.41011628136399991e-03
            -9.05253162176684056e-01 -3.92522295174929370e-01 1.37341746888800008e-03
            -7.63543916101214304e-01 -6.25770048864424111e-01 1.34371539877700002e-03
            -9.38798949431799978e-01 -3.07286607845038640e-01 1.28191954890700000e-03
            -7.01761249508369822e-01 -6.96043134758961735e-01 1.24608003992800003e-03
            -9.65095662701151880e-01 -2.18042040249806945e-01 1.14558934658899993e-03
            -9.83209118931755888e-01 -1.23791243071379786e-01 1.00978328234600009e-03
            -6.36793081128479432e-01 -7.59500496713096385e-01 9.20176790160000019e-04
            -9.91674495391106414e-01 -2.26365158155490721e-02 9.29646549994999965e-04
            -5.62875847282320430e-01 -8.17384200397120120e-01 8.56589133727000008e-04
            -8.52040712508415066e-01 -5.18415014241330518e-01 6.15386714708999983e-04
            -8.95628071786307878e-01 -4.38934044180895544e-01 6.03871648948999971e-04
            -8.01054327269182020e-01 -5.94386253512860452e-01 5.94845518337000045e-04
            -9.32000590079671509e-01 -3.55687518702678673e-01 5.70125663872000010e-04
            -7.42479936307709143e-01 -6.66372334417594137e-01 5.59214403358000039e-04
            -9.61058159177462845e-01 -2.68430041119814777e-01 5.16461052757000009e-04
            -6.77366201148239755e-01 -7.33007040887932448e-01 4.69050144477000015e-04
            -9.82484501226762119e-01 -1.76200902451200042e-01 4.57922790376999998e-04
            -9.95289510146307288e-01 -7.82347486599257042e-02 4.21940731861999974e-04
            -5.19303001975390033e-01 -8.52871980961562470e-01 3.86169486359000012e-04
            -6.03593072093222105e-01 -7.95455299874740396e-01 3.78486748618999996e-04
            -5.36865921109190500e-02 -9.29879052213381518e-02 9.94626755579999973e-03
            -4.86553374082120378e-06 -1.85266504132140825e-01 9.79444233160300069e-03
            -1.06765418975758142e-01 -1.84905183050931066e-01 9.71896770741999955e-03
            -5.30836941417924882e-02 -2.75770484764704726e-01 9.49419737937499952e-03
            5.30504033627839447e-02 -2.75781204716410988e-01 9.49428188197500023e-03
            -1.58622085671435376e-01 -2.74687054609276560e-01 9.34589790786599997e-03
            -2.14445204494488201e-05 -3.64186095585984693e-01 9.12531594206300069e-03
            -1.04914988430363107e-01 -3.63443918544332500e-01 9.05219022291799992e-03
            1.04864081673626888e-01 -3.63479588459438341e-01 9.05280088815500067e-03
            -2.08627922629486762e-01 -3.61280626609999211e-01 8.83561466822400075e-03
            -5.18120519746599964e-02 -4.49088917085259665e-01 8.61991709752400054e-03
            5.17951935597511029e-02 -4.49120375892346768e-01 8.62113246893100003e-03
            -1.54843315178075924e-01 -4.47248516374780913e-01 8.47821467384500017e-03
            1.54827193905627408e-01 -4.47307278943914477e-01 8.48050352790899985e-03
            -2.56140276017512269e-01 -4.43646231602094876e-01 8.20004925499000052e-03
            2.34226618063341104e-05 -5.30618877171519676e-01 8.05948340522799969e-03
            -1.01649251774746777e-01 -5.29472807763872844e-01 7.98860777127399953e-03
            1.01717134574398960e-01 -5.29544754171835552e-01 7.99399700476799945e-03
            -2.02194261916921358e-01 -5.26168629791116338e-01 7.78510859607499991e-03
            2.02323862949432914e-01 -5.26210444351426276e-01 7.79153804155899993e-03
            -3.00510710141570603e-01 -5.20754058639933848e-01 7.45480038135800001e-03
            -4.97960330376871138e-02 -6.06984483724192225e-01 7.37816930418999997e-03
            4.98433454587168742e-02 -6.07038952499467821e-01 7.38469744659799986e-03
            -1.48841216866752446e-01 -6.04351758160328600e-01 7.24375472062699973e-03
            1.49029224279256800e-01 -6.04440914976545063e-01 7.26025998860699989e-03
            -2.46265357891512493e-01 -5.99215148364188877e-01 6.98810701843300026e-03
            2.46700664630586292e-01 -5.99149603820681276e-01 7.00503164311299959e-03
            -3.74309123740985093e-04 -6.78827212554106163e-01 6.66147346365399996e-03
            -3.41072925072044120e-01 -5.91588434909906868e-01 6.62018814704300011e-03
            -9.71404228429712802e-02 -6.77258305225204138e-01 6.58726499455100038e-03
            9.65761382949820724e-02 -6.77382265542601725e-01 6.61340786323900021e-03
            -1.92768914505924621e-01 -6.72769795848620511e-01 6.39638591147699987e-03
            1.92722766075352075e-01 -6.72844154059754507e-01 6.44249850973500036e-03
            2.87096752946218836e-01 -6.65119155289531250e-01 6.14737976983999960e-03
            -2.86354733662800121e-01 -6.65453061962413717e-01 6.10376872736300043e-03
            4.42941798515110591e-02 -7.44105690504613504e-01 5.86173344265100010e-03
            -4.94928688966623856e-02 -7.43955495309057135e-01 5.83808778909299976e-03
            1.37987216855826100e-01 -7.40832713233142259e-01 5.77278755611299998e-03
            -1.42441625829480639e-01 -7.40447788657928041e-01 5.69636220519499983e-03
            -3.77222093379217693e-01 -6.55068074422426982e-01 5.72252496195499970e-03
            2.30718753881372501e-01 -7.33976625328573773e-01 5.57219102781500008e-03
            -2.33632559286769237e-01 -7.33768359340376874e-01 5.45422707722499969e-03
            3.21705619200401083e-01 -7.23286752281953715e-01 5.24677651235799965e-03
            -3.22464223101293912e-01 -7.23913694878484670e-01 5.15340411745500004e-03
            -7.82859132166093680e-03 -8.03633564949003287e-01 5.05634317520200008e-03
            8.27439092097161238e-02 -8.02107784641510002e-01 5.01704054771999995e-03
            -9.81849059419382664e-02 -8.01361752691451823e-01 4.97901306948099996e-03
            1.72817547834930085e-01 -7.96782451531078739e-01 4.89893369505600002e-03
            -1.87067373808409987e-01 -7.95416672008724346e-01 4.76094733620200002e-03
            -4.09614916068560098e-01 -7.09325372579443503e-01 4.78188052084799978e-03
            2.62067262343252616e-01 -7.87153131517981608e-01 4.67690542486600032e-03
            -2.73041452784438310e-01 -7.86280581996435024e-01 4.46239325266899970e-03
            3.49426493526330428e-01 -7.73311808866540162e-01 4.31356067553300034e-03
            -3.56904635055462105e-01 -7.72510679438801584e-01 4.15453096751899984e-03
            4.38528570626996947e-02 -8.55329523637320510e-01 4.25109942447000032e-03
            -4.50765834135742824e-02 -8.55320001521342022e-01 4.35357824810699966e-03
            1.30069045353509632e-01 -8.51332455258564602e-01 4.05420774068400002e-03
            -1.34765978453788321e-01 -8.50742766388000948e-01 4.24883965064200026e-03
            2.13918001871701013e-01 -8.43082127512339352e-01 3.85928354023799998e-03
            -2.22291942972522083e-01 -8.42268441725416150e-01 3.97971797841099959e-03
            -4.40076737466302714e-01 -7.52758912030966898e-01 3.78841810271199992e-03
            2.96826284235199245e-01 -8.30206352595415198e-01 3.64448284283200004e-03
            -3.06620680886550256e-01 -8.29319910906975655e-01 3.64139337464400000e-03
            3.77727450341978965e-01 -8.13827485137954554e-01 3.28751680285900008e-03
            -3.88629305651292056e-01 -8.09813220268305267e-01 3.22961125591499998e-03
            -1.74957108569568504e-02 -9.00648259465546719e-01 2.57245090472699991e-03
            4.27623484086402522e-02 -8.99933883091253439e-01 2.73445598490699993e-03
            -9.23006108103811840e-02 -8.97756978882648937e-01 3.38666652033900017e-03
            1.20902814908180556e-01 -8.96070634151918677e-01 3.43631424158299991e-03
            -1.78593011911284183e-01 -8.90301724212226864e-01 3.43065871148700002e-03
            2.08361295767783478e-01 -8.86329572250103892e-01 3.39408159544900018e-03
            -4.62953438343667867e-01 -7.90804138086501296e-01 2.83115555419499992e-03
            -2.65213554420172637e-01 -8.77646169651378760e-01 3.19234771544299988e-03
            2.94824867508189925e-01 -8.70951794176408445e-01 3.06769483519199991e-03
            -3.50610309516602991e-01 -8.57638555692545212e-01 2.86132808750100016e-03
            3.75508950831214605e-01 -8.51814235868617708e-01 2.52106724557500010e-03
            -3.73670562146501198e-02 -9.36960067852229073e-01 3.04549503190999988e-03
            5.60554568129368186e-02 -9.36354586982625192e-01 3.05234104169899980e-03
            -4.27587080138751574e-01 -8.36571927908680735e-01 2.28570211332600019e-03
            -1.29752811668748663e-01 -9.31146662294305427e-01 2.92610580844900004e-03
            1.49014714619552624e-01 -9.29263995719387670e-01 2.91222693355400013e-03
            -2.20094708665835748e-01 -9.19338479947096499e-01 2.69843874117499990e-03
            2.39751871076822332e-01 -9.15291828174753186e-01 2.65761800614299978e-03
            4.38809265045324637e-01 -8.38596199321462898e-01 1.99917921561299991e-03
            -3.09094132349507500e-01 -8.99971164086340369e-01 2.44583196925200015e-03
            3.26555043317495963e-01 -8.95014950144224497e-01 2.26589034185999995e-03
            -3.93259464932598890e-01 -8.75240908702982323e-01 1.99056338398399986e-03
            3.98557428953685344e-01 -8.78393582644358184e-01 1.76934970702599995e-03
            -4.77524801436366275e-01 -8.38758685765098821e-01 1.76460618128099992e-03
            1.05619774997133802e-02 -9.66058106814914330e-01 2.24662934487400000e-03
            -8.12619770442062905e-02 -9.63278713382986540e-01 2.19547006801600010e-03
            1.02885618686509431e-01 -9.61686245796034012e-01 2.20066282864300007e-03
            -1.71588560214990971e-01 -9.53626729975423970e-01 2.05677144082100014e-03
            1.94875972953508059e-01 -9.49819877933324119e-01 2.06229745077100022e-03
            -2.61212267192195535e-01 -9.36470910107523480e-01 1.88319400533199996e-03
            2.85237793976776566e-01 -9.29992847842621817e-01 1.82986657465800009e-03
            -3.48888761345868770e-01 -9.11794847421221633e-01 1.62690199042700000e-03
            3.64871544778308077e-01 -9.09291535652053118e-01 1.33861458188700008e-03
            -4.36681688778372268e-01 -8.77116659233459162e-01 1.42019616970100009e-03
            4.53634404285341741e-01 -8.70030766343688144e-01 1.39036118836199994e-03
            -2.21561682179156998e-02 -9.85886951405470002e-01 1.42178717363199993e-03
            6.89110227316632046e-02 -9.83960468012635037e-01 1.41011628136399991e-03
            -1.12692301915079562e-01 -9.80233382888667348e-01 1.37341746888800008e-03
            1.60160801193412944e-01 -9.74133452680917888e-01 1.34371539877700002e-03
            -2.03281466079349854e-01 -9.66667043176600793e-01 1.28191954890700000e-03
            2.51910412076831014e-01 -9.55764636845239268e-01 1.24608003992800003e-03
            -2.93717885401254375e-01 -9.44818381106278804e-01 1.14558934658899993e-03
            -3.84398198200009034e-01 -9.13379695763105848e-01 1.00978328234600009e-03
            3.39350183776200964e-01 -9.31229233567976489e-01 9.20176790160000019e-04
            -4.76233449946120158e-01 -8.70133563201586524e-01 9.29646549994999965e-04
            4.26437558554775886e-01 -8.96156883121739800e-01 8.56589133727000008e-04
            2.29402157820559199e-02 -9.97096409211546186e-01 6.15386714708999983e-04
            -6.76860030466576029e-02 -9.95103684599863203e-01 6.03871648948999971e-04
            1.14226431567803308e-01 -9.90926523982995522e-01 5.94845518337000045e-04
            -1.57965868034263740e-01 -9.84979946702421949e-01 5.70125663872000010e-04
            2.05855401830921103e-01 -9.76192653851525205e-01 5.59214403358000039e-04
            -2.48061844840070739e-01 -9.66515800921898927e-01 5.16461052757000009e-04
            2.96119617987687989e-01 -9.53119858303301926e-01 4.69050144477000015e-04
            -3.38647792920898416e-01 -9.38956988112459423e-01 4.57922790376999998e-04
            -4.29891475274967294e-01 -9.01063374236834802e-01 4.21940731861999974e-04
            4.78957300700975919e-01 -8.76165582452989700e-01 3.86169486359000012e-04
            3.87087961219882426e-01 -9.20454583918392855e-01 3.78486748618999996e-04
            5.36865921109189320e-02 -9.29879052213382212e-02 9.94626755579999973e-03
            1.60443066281898195e-01 -9.26374657418929798e-02 9.79444233160300069e-03
            1.06749876325638982e-01 -1.84914156604161273e-01 9.71896770741999955e-03
            2.12282398349287443e-01 -1.83857070035867903e-01 9.49419737937499952e-03
            2.65358730852080726e-01 -9.19476053650232911e-02 9.49428188197500023e-03
            1.58574924546639101e-01 -2.74714283097373002e-01 9.34589790786599997e-03
            3.15383688222305780e-01 -1.82111619292473659e-01 9.12531594206300069e-03
            2.62294172095172529e-01 -2.72581004490611234e-01 9.05219022291799992e-03
            3.67214598199800013e-01 -9.09248355558325144e-02 9.05280088815500067e-03
            2.08564239224676040e-01 -3.61317394240909584e-01 8.83561466822400075e-03
            3.63016384766548184e-01 -2.69415011774885194e-01 8.61991709752400054e-03
            4.14847251659863969e-01 -1.79704234529496909e-01 8.62113246893100003e-03
            3.09906919396422675e-01 -3.57722502737804871e-01 8.47821467384500017e-03
            4.64793063815935714e-01 -8.95693563530248343e-02 8.48050352790899985e-03
            2.56138768851892518e-01 -4.43647101764571172e-01 8.20004925499000052e-03
            4.59541138689013806e-01 -2.65289153965611424e-01 8.05948340522799969e-03
            4.07712276249214967e-01 -3.52767238194547694e-01 7.98860777127399953e-03
            5.09457776840794629e-01 -1.76682754544328025e-01 7.99399700476799945e-03
            3.54578269115095546e-01 -4.38189682215056664e-01 7.78510859607499991e-03
            5.56873544019749311e-01 -8.78876170697032721e-02 7.79153804155899993e-03
            3.00730888835248511e-01 -5.20626938411869133e-01 7.45480038135800001e-03
            5.00765966089288916e-01 -3.46616871480422528e-01 7.37816930418999997e-03
            5.50632826680592657e-01 -2.60353872872881531e-01 7.38469744659799986e-03
            4.48963366955257626e-01 -4.31076154016960977e-01 7.24375472062699973e-03
            5.97975799596026314e-01 -1.73157363356147675e-01 7.26025998860699989e-03
            3.95802861870092559e-01 -5.12879630188210922e-01 6.98810701843300026e-03
            6.42229109891385042e-01 -8.59257592097479594e-02 7.00503164311299959e-03
            5.87694456290164147e-01 -3.39737767487081233e-01 6.66147346365399996e-03
            3.41794150681033970e-01 -5.91172035110410277e-01 6.62018814704300011e-03
            5.37952685827536192e-01 -4.22755226528977568e-01 6.58726499455100038e-03
            6.34918319180440394e-01 -2.55053743608447459e-01 6.61340786323900021e-03
            4.86251276850813374e-01 -5.03327674946391634e-01 6.39638591147699987e-03
            6.79061513241273840e-01 -1.69519265721016804e-01 6.44249850973500036e-03
            7.19558461497490476e-01 -8.39264962493155020e-02 6.14737976983999960e-03
            4.33121889854190145e-01 -5.80717004827118854e-01 6.10376872736300043e-03
            6.66561521003311785e-01 -3.33692960261101512e-01 5.86173344265100010e-03
            6.19537923774347021e-01 -4.14839829425211137e-01 5.83808778909299976e-03
            7.10573558042366216e-01 -2.50915921421913657e-01 5.77278755611299998e-03
            5.70025782239036327e-01 -4.93581960853652135e-01 5.69636220519499983e-03
            3.78694546968368018e-01 -6.54217952946361869e-01 5.72252496195499970e-03
            7.51001780259203944e-01 -1.67180010673528973e-01 5.57219102781500008e-03
            5.18645760038610071e-01 -5.69215911163704735e-01 5.45422707722499969e-03
            7.87237511297114723e-01 -8.30381373732269196e-02 5.24677651235799965e-03
            4.65695538361577510e-01 -6.41219056456575953e-01 5.15340411745500004e-03
            6.92052786918857854e-01 -4.08596541434906591e-01 5.05634317520200008e-03
            7.36017672677663159e-01 -3.29395564936707907e-01 5.01704054771999995e-03
            6.44907182481050723e-01 -4.85711499159630411e-01 4.97901306948099996e-03
            7.76442618233022297e-01 -2.48726839120757742e-01 4.89893369505600002e-03
            5.95317357649024714e-01 -5.59713433941685046e-01 4.76094733620200002e-03
            4.09486334168379595e-01 -7.09399609374125606e-01 4.78188052084799978e-03
            8.12728239734671565e-01 -1.66619659069493303e-01 4.67690542486600032e-03
            5.44418232119106671e-01 -6.29601125395750638e-01 4.46239325266899970e-03
            8.44420918288085187e-01 -8.40436842841495169e-02 4.31356067553300034e-03
            4.90561555561047857e-01 -6.95343820405845192e-01 4.15453096751899984e-03
            7.62663524608111820e-01 -3.89687073573834752e-01 4.25109942447000032e-03
            7.18190557975639576e-01 -4.66697467112634823e-01 4.35357824810699966e-03
            8.02310055996850680e-01 -3.13023130107152880e-01 4.05420774068400002e-03
            6.69381858550964548e-01 -5.42082144100847763e-01 4.24883965064200026e-03
            8.37089540838167778e-01 -2.36282639808469835e-01 3.85928354023799998e-03
            6.18279895853882100e-01 -6.13644690533514114e-01 3.97971797841099959e-03
            4.31869972010801240e-01 -7.57497090275876861e-01 3.78841810271199992e-03
            8.67392933848449954e-01 -1.58044073639084875e-01 3.64448284283200004e-03
            5.64901770266412973e-01 -6.80201254426922253e-01 3.64139337464400000e-03
            8.93659001598460789e-01 -7.97921748660987556e-02 3.28751680285900008e-03
            5.07004168247189413e-01 -7.41469461483279080e-01 3.22961125591499998e-03
            7.71236417142923458e-01 -4.65475859792165436e-01 2.57245090472699991e-03
            8.00746778687720484e-01 -4.12933661498263382e-01 2.73445598490699993e-03
            7.31330044731952955e-01 -5.28813163187935364e-01 3.38666652033900017e-03
            8.36471340214883674e-01 -3.43330407976427388e-01 3.43631424158299991e-03
            6.81727404245233370e-01 -5.99816947359662622e-01 3.43065871148700002e-03
            8.71764573577876400e-01 -2.62718610824709486e-01 3.39408159544900018e-03
            4.53379753828933085e-01 -7.96331507418220008e-01 2.83115555419499992e-03
            6.27457101242114734e-01 -6.68504760381525842e-01 3.19234771544299988e-03
            9.01678812982500411e-01 -1.80150072158730717e-01 3.06769483519199991e-03
            5.67431621736437464e-01 -7.32456712716375913e-01 2.86132808750100016e-03
            9.25447242983060026e-01 -1.00706827166035656e-01 2.52106724557500010e-03
            7.92747692984296481e-01 -5.00840853872642877e-01 3.04549503190999988e-03
            8.38934587683507482e-01 -4.19631843870568122e-01 3.05234104169899980e-03
            5.10699001592465396e-01 -7.88587237684511999e-01 2.28570211332600019e-03
            7.41520258361583751e-01 -5.77942562264747273e-01 2.92610580844900004e-03
            8.79273584424999810e-01 -3.35581469461473214e-01 2.91222693355400013e-03
            6.86123123977838145e-01 -6.50276848916697348e-01 2.69843874117499990e-03
            9.12541910614048657e-01 -2.50014703129997129e-01 2.65761800614299978e-03
            9.45650244652127658e-01 -3.92781287155025416e-02 1.99917921561299991e-03
            6.24850824597470211e-01 -7.17668952818553363e-01 2.44583196925200015e-03
            9.38383205250509222e-01 -1.64702511825233294e-01 2.26589034185999995e-03
            5.61351128901859453e-01 -7.78193141261797616e-01 1.99056338398399986e-03
            9.59989871568082576e-01 -9.40359329812763545e-02 1.76934970702599995e-03
            4.87623928799241435e-01 -8.32927951863562610e-01 1.76460618128099992e-03
            8.41911850783473525e-01 -4.73882112578505266e-01 2.24662934487400000e-03
            7.93592848192352096e-01 -5.52014293173524129e-01 2.19547006801600010e-03
            8.84287528672705880e-01 -3.91741563431421191e-01 2.20066282864300007e-03
            7.40070693779104616e-01 -6.25413417132690275e-01 2.05677144082100014e-03
            9.20006129786447135e-01 -3.06142395801715150e-01 2.06229745077100022e-03
            6.80401464462150773e-01 -6.94451914222331657e-01 1.88319400533199996e-03
            9.48016328557934718e-01 -2.17973248217990828e-01 1.82986657465800009e-03
            6.15193120233599400e-01 -7.58043954131019659e-01 1.62690199042700000e-03
            9.69905341709995428e-01 -1.38657740929940709e-01 1.33861458188700008e-03
            5.41264464589528038e-01 -8.16735765466290231e-01 1.42019616970100009e-03
            9.80285947870348062e-01 -4.21564650301180371e-02 1.39036118836199994e-03
            8.42725061067773340e-01 -5.12131280229971675e-01 1.42178717363199993e-03
            8.86590272984399297e-01 -4.32301537719929796e-01 1.41011628136399991e-03
            7.92560860261604438e-01 -5.87711087713738589e-01 1.37341746888800008e-03
            9.23704717294627886e-01 -3.48363403816493777e-01 1.34371539877700002e-03
            7.35517483352449375e-01 -6.59380435331562875e-01 1.28191954890700000e-03
            9.53671661585201003e-01 -2.59721502086277978e-01 1.24608003992800003e-03
            6.71377777299897338e-01 -7.26776340856472358e-01 1.14558934658899993e-03
            5.98810920731746688e-01 -7.89588452691726617e-01 1.00978328234600009e-03
            9.76143264904680619e-01 -1.71728736854880687e-01 9.20176790160000019e-04
            5.15441045444986035e-01 -8.47497047386038371e-01 9.29646549994999965e-04
            9.89313405837096593e-01 -7.87726827246202216e-02 8.56589133727000008e-04
            8.74980928290471027e-01 -4.78681394970216278e-01 6.15386714708999983e-04
            8.27942068739650261e-01 -5.56169640418968325e-01 6.03871648948999971e-04
            9.15280758836985453e-01 -3.96540270470135681e-01 5.94845518337000045e-04
            7.74034722045407797e-01 -6.29292427999743831e-01 5.70125663872000010e-04
            9.48335338138630246e-01 -3.09820319433931568e-01 5.59214403358000039e-04
            7.12996314337391968e-01 -6.98085759802084760e-01 5.16461052757000009e-04
            9.73485819135927799e-01 -2.20112817415370088e-01 4.69050144477000015e-04
            6.43836708305863592e-01 -7.62756085661259853e-01 4.57922790376999998e-04
            5.65398034871339661e-01 -8.22828625576909056e-01 4.21940731861999974e-04
            9.98260302676366229e-01 -2.32936014914277394e-02 3.86169486359000012e-04
            9.90681033313104864e-01 -1.24999284043652931e-01 3.78486748618999996e-04
            ];




    case 77

        xyw=[
            -0.00000000000000000e+00 0.00000000000000000e+00 7.22136262576800018e-03
            9.11898994730129947e-02 -0.00000000000000000e+00 7.18170208769999992e-03
            1.36420069879017641e-01 7.87365002628742788e-02 7.10267554837199985e-03
            1.81626070716434190e-01 -3.98786798956569591e-05 7.06329804647600010e-03
            2.26118199399790692e-01 7.82399590881715651e-02 6.94577127469500014e-03
            1.80940984854166292e-01 1.56577233696842233e-01 6.94583072076000035e-03
            2.70552830951540868e-01 -1.37231362572192729e-04 6.86785783981600038e-03
            2.69527211420165946e-01 1.55430528676271162e-01 6.75208334917300028e-03
            3.13917219890731403e-01 7.74929017913242035e-02 6.71322342063399979e-03
            2.24385979273877462e-01 2.32881196612900937e-01 6.71356749662900022e-03
            3.57211744280805876e-01 -2.68355751715055344e-04 6.59815299243199978e-03
            3.55814986776059772e-01 1.53881541615999318e-01 6.48498667156699980e-03
            3.11481039386652891e-01 2.30925814006707336e-01 6.48557078261800032e-03
            3.99036742817477030e-01 7.65556481996069321e-02 6.40829273834399998e-03
            2.66346135906748660e-01 3.07036989058170229e-01 6.40939593785600035e-03
            4.40843902165012680e-01 -3.47722372532707443e-04 6.25791916856600038e-03
            3.95866810489687249e-01 2.28472097676950037e-01 6.18785584192700026e-03
            4.39001230694778366e-01 1.52036761901387957e-01 6.14860221255199985e-03
            3.51554930541187305e-01 3.04166877166661020e-01 6.15087493594800028e-03
            4.80692801711543272e-01 7.55563155775556966e-02 6.03512873584500023e-03
            3.06345338760585728e-01 3.78484698244239970e-01 6.03790924661300014e-03
            5.20699075573833436e-01 -1.97485943777200173e-04 5.85171462448200001e-03
            4.76722885716839961e-01 2.25674045654640054e-01 5.82589813729699965e-03
            4.33627668249923492e-01 3.00773303063882946e-01 5.82831101447100000e-03
            5.18285548666786444e-01 1.50060176243171334e-01 5.74797153569299966e-03
            3.89236264616326988e-01 3.74673979352484032e-01 5.75425064206199974e-03
            5.58115638081190690e-01 7.47035904385695143e-02 5.59837142076200032e-03
            3.43803187296567003e-01 4.46755332440501418e-01 5.60506991703199967e-03
            5.11762673749872610e-01 2.97007740037343426e-01 5.44938025848299996e-03
            5.96060760224002895e-01 4.95272286968001930e-04 5.38478976123000011e-03
            5.53275604683822975e-01 2.22667346649420700e-01 5.40646093270800033e-03
            4.68561113125876505e-01 3.70366192362330482e-01 5.41531091663099984e-03
            5.92923818427177562e-01 1.48110941077455954e-01 5.28847197034399991e-03
            4.23888425861630724e-01 4.42073212609056132e-01 5.30474416620400034e-03
            6.30608102403852677e-01 7.42586907151157882e-02 5.10231770740100007e-03
            3.77994075730522683e-01 5.11519029868972241e-01 5.11945714409100042e-03
            5.85330590878151646e-01 2.92783903596042505e-01 5.02236053980400013e-03
            5.43658400426314747e-01 3.65518009493415230e-01 5.02983271602400021e-03
            6.24924247646848841e-01 2.19350604023766466e-01 4.93660000075799976e-03
            5.00072283672651441e-01 4.36813729360414904e-01 4.96045900220599967e-03
            6.66306776824463154e-01 2.19843680677003035e-03 4.86371774827700030e-03
            6.62354236699173038e-01 1.46149996843528029e-01 4.77474165960099957e-03
            4.54775624331602957e-01 5.06030124907866918e-01 4.81565372781499971e-03
            6.97679597210121072e-01 7.44065327855273489e-02 4.55063054068700033e-03
            4.08098308027534884e-01 5.72560120968358288e-01 4.59544679538400041e-03
            6.14105777200499525e-01 3.59726778661403235e-01 4.60203789027800042e-03
            6.53909258624762124e-01 2.87684656774637115e-01 4.55465787132300001e-03
            5.71963977310523286e-01 4.30543995872102403e-01 4.57230938209900011e-03
            6.91279923249720207e-01 2.15292879208524801e-01 4.42444979191600023e-03
            5.27619621557057439e-01 4.99485974586054371e-01 4.47071236481500021e-03
            7.31012790770115495e-01 5.40724557881489334e-03 4.30104406947099966e-03
            7.26271906327262018e-01 1.43862662978291561e-01 4.21295726809199980e-03
            4.81291506686956139e-01 5.65991123735779533e-01 4.29875887114499963e-03
            7.59158696785791287e-01 7.52203723476614766e-02 3.95316724490200021e-03
            4.33610912610090438e-01 6.29474181605447103e-01 4.05258060727699959e-03
            6.78906938049608333e-01 3.53639407588749177e-01 4.13728997613100041e-03
            6.38598965188012069e-01 4.23803772646022148e-01 4.13336894192800012e-03
            7.16559870454215320e-01 2.82315013504101986e-01 4.05842545064200033e-03
            5.95745864531659008e-01 4.92094943413269981e-01 4.06045140728300037e-03
            7.51533832408555313e-01 2.11044020343923305e-01 3.88425009578400020e-03
            5.50473296536106327e-01 5.58042030714757553e-01 3.92810303565400021e-03
            7.89925116265259186e-01 1.02485593615964706e-02 3.72535039921600020e-03
            7.84111102648339164e-01 1.41987705935640501e-01 3.61910327768300011e-03
            5.03175101219821119e-01 6.21246209108857972e-01 3.73821698265100000e-03
            6.97191694171157672e-01 4.20423896950354703e-01 3.70036558101400010e-03
            7.35914939693704473e-01 3.50300737365393522e-01 3.68592864226299994e-03
            6.55678291944845060e-01 4.88347353913775706e-01 3.62603685967500004e-03
            8.14704385691526922e-01 7.78001307804098291e-02 3.35280890303199996e-03
            4.54918461366089655e-01 6.81238459880157299e-01 3.48157001957200007e-03
            7.71584691231618325e-01 2.79126759100493060e-01 3.56583693490600002e-03
            6.11742624803793378e-01 5.53340570010183996e-01 3.48356058743999998e-03
            8.04270939478598623e-01 2.08941437509632622e-01 3.32565976192399993e-03
            5.65893272572581929e-01 6.14985560630172068e-01 3.29489471807600017e-03
            8.42165714326061576e-01 1.63378765396783587e-02 3.16058812290699988e-03
            8.34769621838661657e-01 1.43587615879625485e-01 3.01445684751700015e-03
            5.18886672165121632e-01 6.73110715965157480e-01 3.06783191591500012e-03
            7.51022559572803727e-01 4.14096539674263409e-01 3.32665344726499994e-03
            7.09793636675365325e-01 4.83954561145078344e-01 3.30307358571299992e-03
            7.88493968017017210e-01 3.42483049890051183e-01 3.25408345210300006e-03
            6.65179004887006542e-01 5.51162387881440785e-01 3.19930874407000016e-03
            8.22147612392106764e-01 2.70590067657131061e-01 3.06215445555199986e-03
            4.71352009201336797e-01 7.27630148643887642e-01 2.79080351623500018e-03
            8.63165104721542620e-01 8.41192131073621036e-02 2.87163083124200016e-03
            6.17903515562940098e-01 6.14972220874635989e-01 3.02750821415299996e-03
            8.52709974258748393e-01 2.02014727081945983e-01 2.72451847997900011e-03
            5.68983104632018510e-01 6.74734635545256700e-01 2.78664812139600010e-03
            8.86281498487202057e-01 2.20719302811990779e-02 2.54475514915600016e-03
            8.82295270078386151e-01 1.43475640074816502e-01 2.57116546832600020e-03
            5.19246553319736681e-01 7.29891604028140906e-01 2.47222339185300009e-03
            7.64769096886661393e-01 4.67071488006187086e-01 2.87470048159199978e-03
            8.04004786242535685e-01 3.96120065765089679e-01 2.86031186141600009e-03
            7.21418837220728970e-01 5.35467908961158146e-01 2.81094637882399988e-03
            8.38814533001302931e-01 3.23525366464308184e-01 2.74075746044699989e-03
            6.74667618043454320e-01 6.00636391253081214e-01 2.68775126324999987e-03
            8.69572260905150829e-01 2.51542998561805509e-01 2.46112952592700013e-03
            9.06048309292831200e-01 8.04477385698380426e-02 2.36556594380100006e-03
            6.25226736391707449e-01 6.61990498122072779e-01 2.49944003710999988e-03
            4.72077140322839872e-01 7.79568540002584798e-01 2.14635104513900008e-03
            5.73501561186594233e-01 7.19164209845148727e-01 2.25515678876800005e-03
            9.00507936200001358e-01 1.91545389494470641e-01 2.29174836455199983e-03
            8.20928345984985763e-01 4.33955017915453156e-01 2.26703879807000014e-03
            7.83872579604646846e-01 4.98273223995181924e-01 2.11694199000600014e-03
            8.55223690306502915e-01 3.64351203803251256e-01 2.27988297223499998e-03
            7.46103575168212574e-01 5.55918014687375783e-01 1.92861219257499990e-03
            5.23027916089144451e-01 7.71259042629548031e-01 1.97013878030600012e-03
            9.31425882664175586e-01 3.26917344467800297e-02 2.04474818264299999e-03
            8.85413012744074357e-01 2.91634390004629318e-01 2.10252607917200008e-03
            7.07190096012797142e-01 6.09174562293896327e-01 1.87858335006200008e-03
            9.25076744174782695e-01 1.27333301402449667e-01 2.07844803552899998e-03
            6.63817352509975200e-01 6.62147040958489197e-01 1.87233680736699991e-03
            6.15683224782216931e-01 7.14670773608313326e-01 1.78430481805599991e-03
            9.16393421483082116e-01 2.31822920469051619e-01 1.91159890891400008e-03
            4.90352176236370230e-01 8.15092131636892203e-01 1.77778576656199990e-03
            5.67109240041191121e-01 7.64278499362628039e-01 1.61669914040900005e-03
            9.49919914916783292e-01 7.69359412585199492e-02 1.75720749155900010e-03
            8.38450789431827270e-01 4.57692146855183168e-01 1.65060830486800001e-03
            8.02831587872182562e-01 5.17929183684352368e-01 1.64584059562500004e-03
            9.40646025088715643e-01 1.67851953493418304e-01 1.68314707775099994e-03
            8.70259610582841736e-01 3.94696213375152483e-01 1.69293087211900011e-03
            8.98556907402872085e-01 3.25345270026698496e-01 1.65361412771100006e-03
            7.66748697704903392e-01 5.71789361797825801e-01 1.13642016279599995e-03
            7.48478440915308574e-01 5.97235943034087668e-01 8.56057631169000006e-04
            7.09849356953799138e-01 6.45505518492381136e-01 1.58908904780400003e-03
            6.59705994017414032e-01 7.01365397061908258e-01 1.51893311022200002e-03
            9.28000702129639032e-01 2.68239385121203266e-01 1.46681544418000008e-03
            6.09186356900383585e-01 7.53000745864821464e-01 1.33401257950899990e-03
            5.38497011383011648e-01 8.05208098651456927e-01 1.42675368759600011e-03
            9.68880232540156539e-01 2.63986912722468624e-02 1.43941250634100010e-03
            9.63995671397759657e-01 1.17490095473076206e-01 1.37476644712400007e-03
            9.51335828019985841e-01 2.04835824993499338e-01 1.26363484330699999e-03
            9.06933010136960926e-01 3.57301880375143210e-01 1.23636098643800001e-03
            8.43890202082035801e-01 4.89675726391281596e-01 1.25540140813499993e-03
            8.06214396089813712e-01 5.50169350070916163e-01 1.17585700181700010e-03
            8.78213067319584928e-01 4.26068922194598187e-01 1.27531648584499994e-03
            7.75566419381766514e-01 5.94095664249923550e-01 6.45429343702000010e-04
            7.46625203970109785e-01 6.30886712247863080e-01 1.03253300932900000e-03
            6.99255112827864012e-01 6.85227572092496295e-01 1.15965694635099996e-03
            9.34128826784312416e-01 3.02134971434106558e-01 9.87631064347000050e-04
            6.47615666559195446e-01 7.38549534618431647e-01 1.02216970241799992e-03
            5.81840664098543292e-01 7.92005617555544772e-01 1.06242630376100007e-03
            5.08936294288077629e-01 8.40919263248634974e-01 1.07433276771499997e-03
            9.81211526461203354e-01 6.84354947286966719e-02 1.04569291981000006e-03
            9.72408880257945474e-01 1.57355593460202170e-01 9.70235686178999969e-04
            9.56318435037960990e-01 2.41313280906473165e-01 8.65312708298999990e-04
            9.10113509206321947e-01 3.86319187283116205e-01 6.78618929186000034e-04
            8.60436218105287809e-01 4.89233004882684619e-01 7.12640554481999981e-04
            8.25703652607938210e-01 5.46646051820743328e-01 7.79849561438999994e-04
            7.83693519308070075e-01 6.05912169168822179e-01 7.99787468057999956e-04
            8.88871142222162347e-01 4.38134265476513463e-01 6.10870396132000043e-04
            7.34609395552191846e-01 6.65215278762726347e-01 7.87863412149000016e-04
            9.32968978868632837e-01 3.37616346851429161e-01 5.71748160716000040e-04
            6.81852467755609704e-01 7.21071211833703263e-01 6.86081209356000045e-04
            6.19806826855616388e-01 7.75621516933603927e-01 6.82048477971000005e-04
            5.50510744420704468e-01 8.26346588851067265e-01 6.89150458834000010e-04
            9.92818447943157878e-01 2.33146843039550647e-02 6.79102356988000038e-04
            9.87346208651238233e-01 1.10614401915643520e-01 6.42916727870000053e-04
            9.74483813191885040e-01 1.98004577921158442e-01 5.76658274574999965e-04
            9.54778656536635184e-01 2.79823633982320241e-01 5.06490665150999991e-04
            9.20861850628062695e-01 3.84719958828251107e-01 3.65614865653999979e-04
            8.54711405086357678e-01 5.15355522372411445e-01 3.75571429357999999e-04
            8.12618879806817573e-01 5.79720994467935458e-01 3.58493698192999981e-04
            7.64858607082260944e-01 6.41432528326842721e-01 3.55000291663000022e-04
            8.90798882887371413e-01 4.50545021079945207e-01 3.28602265489999989e-04
            7.12503228605111505e-01 6.99482150258931457e-01 3.13506391891999988e-04
            6.53252276406545151e-01 7.55307844991842559e-01 2.95303559031000000e-04
            5.87441927391727958e-01 8.07589568085765430e-01 2.94712962742999993e-04
            5.16558910482619038e-01 8.54689159707880619e-01 2.92691680609000020e-04
            9.96569298591459263e-01 6.54233637247941535e-02 2.82028007154999985e-04
            9.87412662150806142e-01 1.50625623955868454e-01 2.57150644410000014e-04
            9.47069271954280367e-01 3.18214713864706789e-01 1.98235288888000004e-04
            9.70854337050417016e-01 2.35886977661771774e-01 2.09409578575000003e-04
            4.55949497365065043e-02 7.89727695121784368e-02 7.18170208769999992e-03
            2.22255067795720473e-05 1.57511496232714737e-01 7.10267554837199985e-03
            9.08475713080761221e-02 1.57272851890033094e-01 7.06329804647600010e-03
            4.53013075384836317e-02 2.34944084482299725e-01 6.94577127469500014e-03
            -4.51293696086750434e-02 2.34988106317904483e-01 6.94583072076000035e-03
            1.35395261321953925e-01 2.34237008988545020e-01 6.86785783981600038e-03
            1.56819352786487785e-04 3.11132676439178524e-01 6.75208334917300028e-03
            8.98477883811063666e-02 3.10606738006421190e-01 6.71322342063399979e-03
            -8.94880426935520112e-02 3.10764556610676923e-01 6.71356749662900022e-03
            1.78838275038639916e-01 3.09220267201471000e-01 6.59815299243199978e-03
            4.46421691750622240e-02 3.85085588403291490e-01 6.48498667156699980e-03
            -4.42471016260823982e-02 3.85213399909376364e-01 6.48557078261800032e-03
            1.33219235264694563e-01 3.83853780423136171e-01 6.40829273834399998e-03
            -1.32728764472485772e-01 3.84181014424152090e-01 6.40939593785600035e-03
            2.20723087490583919e-01 3.81608157192096287e-01 6.25791916856600038e-03
            7.07646006852511215e-05 4.57066763237664264e-01 6.18785584192700026e-03
            8.78329172316611312e-02 4.56204599025004875e-01 6.14860221255199985e-03
            -8.76387773455156310e-02 4.56538939257672471e-01 6.15087493594800028e-03
            1.74912712149254551e-01 4.54070335487290200e-01 6.03512873584500023e-03
            -1.74604694242906477e-01 4.54545194819736875e-01 6.03790924661300014e-03
            2.60520565631118139e-01 4.50839884202124452e-01 5.85171462448200001e-03
            4.29219863466925727e-02 5.25691152423529151e-01 5.82589813729699965e-03
            -4.36634871085168422e-02 5.25919228020186091e-01 5.82831101447100000e-03
            1.29186849610436777e-01 5.23878539681378652e-01 5.74797153569299966e-03
            -1.29859051948093857e-01 5.24425482908143281e-01 5.75425064206199974e-03
            2.14362611966885858e-01 5.20694116046957478e-01 5.59837142076200032e-03
            -2.14999873521352858e-01 5.21119960321137166e-01 5.60506991703199967e-03
            -1.33491111800764406e-03 5.91703346194709101e-01 5.44938025848299996e-03
            2.97601461729696870e-01 5.16451396696535503e-01 5.38478976123000011e-03
            8.38022235502374480e-02 5.90484402275097575e-01 5.40646093270800033e-03
            -8.64659747257539302e-02 5.90968923373688559e-01 5.41531091663099984e-03
            1.68194071662091854e-01 5.87542559805535602e-01 5.28847197034399991e-03
            -1.70902419521226423e-01 5.88134751470896888e-01 5.30474416620400034e-03
            2.50994138590864502e-01 5.83251981871593106e-01 5.10231770740100007e-03
            -2.53991436520439651e-01 5.83111986997137843e-01 5.11945714409100042e-03
            3.91069971057289509e-02 6.53303113110656586e-01 5.02236053980400013e-03
            -4.47186815488617440e-02 6.53580990496708925e-01 5.02983271602400021e-03
            1.22498928403381585e-01 6.50875575914932059e-01 4.93660000075799976e-03
            -1.28255644511613920e-01 6.51482166069221824e-01 4.96045900220599967e-03
            3.31249486288954009e-01 5.78137813847098614e-01 4.86371774827700030e-03
            2.04607508320075765e-01 6.46690593707499040e-01 4.77474165960099957e-03
            -2.10847131084623718e-01 6.46862306147030108e-01 4.81565372781499971e-03
            2.84401851005274264e-01 6.41411521278823171e-01 4.55063054068700033e-03
            -2.91802455938722005e-01 6.39703562477471288e-01 4.59544679538400041e-03
            -4.47964014206716620e-03 7.11694592997120679e-01 4.60203789027800042e-03
            7.78124082665383326e-02 7.10144358126211017e-01 4.55465787132300001e-03
            -8.68800492168414207e-02 7.10607332336550712e-01 4.57230938209900011e-03
            1.59190858976383187e-01 7.06312414264677058e-01 4.42444979191600023e-03
            -1.68757732047022824e-01 7.06674983096570619e-01 4.47071236481500021e-03
            3.60823583349303034e-01 6.35779270087686088e-01 4.30104406947099966e-03
            2.38547232368351458e-01 7.00901252423506982e-01 4.21295726809199980e-03
            -2.49516938128208438e-01 6.99806233284481882e-01 4.29875887114499963e-03
            3.14436595057696389e-01 6.95060903094213756e-01 3.95316724490200021e-03
            -3.28335175991690986e-01 6.90255156481216159e-01 4.05258060727699959e-03
            3.31927582736680979e-02 7.64770358950843554e-01 4.13728997613100041e-03
            -4.77253507371336524e-02 7.64944813006283919e-01 4.13336894192800012e-03
            1.13787961662808523e-01 7.61716557897887947e-01 4.05842545064200033e-03
            -1.28293789803927971e-01 7.61978524590574469e-01 4.06045140728300037e-03
            1.92997433269640278e-01 7.56369400841247375e-01 3.88425009578400020e-03
            -2.08041926710382574e-01 7.55744874262611388e-01 3.92810303565400021e-03
            3.86087045373294357e-01 6.89219497453888885e-01 3.72535039921600020e-03
            2.69090590958830478e-01 7.50053987250709486e-01 3.61910327768300011e-03
            -2.86427448483139835e-01 7.46385524762600339e-01 3.73821698265100000e-03
            -1.55019280314791334e-02 8.13997666934911046e-01 3.70036558101400010e-03
            6.45881323240009297e-02 8.12471401481937971e-01 3.68592864226299994e-03
            -9.50820683878171591e-02 8.12007734491113298e-01 3.62603685967500004e-03
            3.39975303172177001e-01 7.44454759973662505e-01 3.35280890303199996e-03
            -3.62510581608157412e-01 7.34590174133642027e-01 3.48157001957200007e-03
            1.44061481358762983e-01 8.07775323328000172e-01 3.56583693490600002e-03
            -1.73335678171484375e-01 8.06454938662949594e-01 3.48356058743999998e-03
            2.21186876952718797e-01 8.00989783868859551e-01 3.32565976192399993e-03
            -2.49646482180053142e-01 7.97570730193653654e-01 3.29489471807600017e-03
            4.06933841035775568e-01 7.37505841072476853e-01 3.16058812290699988e-03
            2.93034287898733370e-01 7.94725506759622835e-01 3.01445684751700015e-03
            -3.23487643502797295e-01 7.85924397762741922e-01 3.06783191591500012e-03
            1.68931568092592667e-02 8.57452885242391716e-01 3.32665344726499994e-03
            -6.42201258913045331e-02 8.56676601377947544e-01 3.30307358571299992e-03
            9.76479624381511035e-02 8.54097331978557195e-01 3.25408345210300006e-03
            -1.44731127072316723e-01 8.51643110236921408e-01 3.19930874407000016e-03
            1.76735933593227978e-01 8.47295751820851817e-01 3.06215445555199986e-03
            -3.94470188684385492e-01 7.72017888415137854e-01 2.79080351623500018e-03
            3.58733176863438807e-01 7.89582514902792298e-01 2.87163083124200016e-03
            -2.23629808117699613e-01 8.42606252002537359e-01 3.02750821415299996e-03
            2.51405101537828857e-01 8.39475863309423809e-01 2.72451847997900011e-03
            -2.99845782879417710e-01 8.30121140708095728e-01 2.78664812139600010e-03
            4.24025896909523747e-01 7.78578257734655943e-01 2.54475514915600016e-03
            3.16894085910169432e-01 8.35827937564142975e-01 2.57116546832600020e-03
            -3.72481394437473856e-01 8.14626508016473516e-01 2.47222339185300009e-03
            -2.21112255534260740e-02 8.95845209936224962e-01 2.87470048159199978e-03
            5.89523532199376574e-02 8.94348602532858084e-01 2.86031186141600009e-03
            -1.03019393461331307e-01 8.92500994282361182e-01 2.81094637882399988e-03
            1.39226080373890559e-01 8.88197377874862748e-01 2.74075746044699989e-03
            -1.82832564240850592e-01 8.84597491962908489e-01 2.68775126324999987e-03
            2.16943503553939404e-01 8.78843167651033208e-01 2.46112952592700013e-03
            3.83354369367926717e-01 8.24884722188451103e-01 2.36556594380100006e-03
            -2.60687220241775819e-01 8.72457485901491525e-01 2.49944003710999988e-03
            -4.39087589471963691e-01 7.98615066066782897e-01 2.14635104513900008e-03
            -3.36063694625164611e-01 8.56249026020200588e-01 2.25515678876800005e-03
            2.84370794820004158e-01 8.75635443805932945e-01 2.29174836455199983e-03
            3.46481033779792794e-02 9.27922311267465161e-01 2.26703879807000014e-03
            -3.95809802030779104e-02 9.27990179265254733e-01 2.11694199000600014e-03
            1.12074446760194610e-01 9.22821043625332460e-01 2.27988297223499998e-03
            -1.08387335556571540e-01 9.24103657293752434e-01 1.92861219257499990e-03
            -4.06415965771081689e-01 8.38584983536408868e-01 1.97013878030600012e-03
            4.37401068807401650e-01 8.22984343352909709e-01 2.04474818264299999e-03
            1.90143716010849700e-01 9.12607356879998055e-01 2.10252607917200008e-03
            -1.73965598279381434e-01 9.17031869598786686e-01 1.87858335006200008e-03
            3.52264498325129460e-01 8.64806611606784847e-01 2.07844803552899998e-03
            -2.41527482255759152e-01 9.05956211225813046e-01 1.87233680736699991e-03
            -3.11081432895968124e-01 8.90532700149481382e-01 1.78430481805599991e-03
            2.57432172435842876e-01 9.09531443099815262e-01 1.91159890891400008e-03
            -4.60714404304173208e-01 8.32203507240126927e-01 1.77778576656199990e-03
            -3.78329975993689172e-01 8.73270258277872746e-01 1.61669914040900005e-03
            4.08331477864446235e-01 8.61122748507946789e-01 1.75720749155900010e-03
            2.28523684266870829e-02 9.54965756898671136e-01 1.65060830486800001e-03
            -4.71240364958946320e-02 9.54237141900085128e-01 1.64584059562500004e-03
            3.24958956744213434e-01 8.98549330442391270e-01 1.68314707775099994e-03
            9.33128577310156776e-02 9.51015037339870029e-01 1.69293087211900011e-03
            1.67521184857207212e-01 9.40845743570217885e-01 1.65361412771100006e-03
            -1.11809764078156737e-01 9.49918531429994495e-01 1.13642016279599995e-03
            -1.42982278263021345e-01 9.46819315534671024e-01 8.56057631169000006e-04
            -2.04099498820547981e-01 9.37500335228228732e-01 1.58908904780400003e-03
            -2.77547254182265113e-01 9.22004848378899511e-01 1.51893311022200002e-03
            2.31698229254339971e-01 9.37791875334664904e-01 1.46681544418000008e-03
            -3.47524596537373665e-01 9.04071233647036676e-01 1.33401257950899990e-03
            -4.28082163073622179e-01 8.68956141045414632e-01 1.42675368759600011e-03
            4.61578179001650102e-01 8.52274240240473269e-01 1.43941250634100010e-03
            3.80248428326136811e-01 8.93589788305233945e-01 1.37476644712400007e-03
            2.98274885960479186e-01 9.26298907092361068e-01 1.26363484330699999e-03
            1.44033999843658084e-01 9.64077966496869609e-01 1.23636098643800001e-03
            -2.12651763042987245e-03 9.75668216203467331e-01 1.25540140813499993e-03
            -7.33534355000802774e-02 9.73286822945966357e-01 1.17585700181700010e-03
            7.01200232762151515e-02 9.73589287331512954e-01 1.27531648584499994e-03
            -1.26718727827740879e-01 9.68708053631707533e-01 6.45429343702000010e-04
            -1.73051317731637622e-01 9.62039749867784622e-01 1.03253300932900000e-03
            -2.43796928391702472e-01 9.48186477481332224e-01 1.15965694635099996e-03
            2.05407852758534348e-01 9.60046780119621523e-01 9.87631064347000050e-04
            -3.15794825653138711e-01 9.30126386438271391e-01 1.02216970241799992e-03
            -3.94976652693812491e-01 8.99891604841919435e-01 1.06242630376100007e-03
            -4.73789297360972805e-01 8.61211391385705904e-01 1.07433276771499997e-03
            4.31338886274994338e-01 8.83971855765857351e-01 1.04569291981000006e-03
            3.49930498764861375e-01 9.20808589899062113e-01 9.70235686178999969e-04
            2.69175785983404581e-01 9.48852699303489144e-01 8.65312708298999990e-04
            1.20494524446624215e-01 9.81341012941635449e-01 6.78618929186000034e-04
            6.52989845444280122e-03 9.89776125656729455e-01 7.12640554481999981e-04
            -6.05575414512590504e-02 9.88403365066447215e-01 7.79849561438999994e-04
            -1.32888571308299147e-01 9.81654581086430222e-01 7.99787468057999956e-04
            6.50001669399853932e-02 9.88852122593540050e-01 6.10870396132000043e-04
            -2.08788632617971953e-01 9.68798037788292654e-01 7.87863412149000016e-04
            1.74100156328080485e-01 9.76783010068777768e-01 5.71748160716000040e-04
            -2.83539753507812464e-01 9.51037164626319376e-01 6.86081209356000045e-04
            -3.61804523958514923e-01 9.24579215962788803e-01 6.82048477971000005e-04
            -4.40381766065286673e-01 8.89929584150146114e-01 6.89150458834000010e-04
            4.76218115083139659e-01 8.71463339416590377e-01 6.79102356988000038e-04
            3.97878222242249779e-01 9.10374099980044926e-01 6.42916727870000053e-04
            3.15764912050603974e-01 9.42930026761480966e-01 5.76658274574999965e-04
            2.35054952660349842e-01 9.66774388543063412e-01 5.06490665150999991e-04
            1.27253667625862765e-01 9.89849735433978917e-01 3.65614865653999979e-04
            -1.89552718119289539e-02 9.97879550895283574e-01 3.75571429357999999e-04
            -9.57436684130013732e-02 9.93609090741525258e-01 3.58493698192999981e-04
            -1.73067560803596587e-01 9.83103248199839741e-01 3.55000291663000022e-04
            5.52160076398578101e-02 9.96726972783235299e-01 3.28602265489999989e-04
            -2.49517697315442599e-01 9.66786971379923643e-01 3.13506391891999988e-04
            -3.27489643237341865e-01 9.43386988944003191e-01 2.95303559031000000e-04
            -4.05672118097711309e-01 9.12534416412212890e-01 2.94712962742999993e-04
            -4.81903069404890305e-01 8.74697718883100328e-01 2.92691680609000020e-04
            4.41626354309028746e-01 8.95766011074240476e-01 2.82028007154999985e-04
            3.63260714268739071e-01 9.30437261418953665e-01 2.57150644410000014e-04
            1.97952609912308009e-01 9.79293405588393395e-01 1.98235288888000004e-04
            2.81143053448181857e-01 9.58728008090846795e-01 2.09409578575000003e-04
            -4.55949497365064765e-02 7.89727695121784506e-02 7.18170208769999992e-03
            -1.36397844372238064e-01 7.87749959698405000e-02 7.10267554837199985e-03
            -9.07784994083579988e-02 1.57312730569928810e-01 7.06329804647600010e-03
            -1.80816891861307061e-01 1.56704125394128202e-01 6.94577127469500014e-03
            -2.26070354462841350e-01 7.84108726210622914e-02 6.94583072076000035e-03
            -1.35157569629586888e-01 2.34374240351117258e-01 6.86785783981600038e-03
            -2.69370392067379449e-01 1.55702147762907445e-01 6.75208334917300028e-03
            -2.24069431509625050e-01 2.33113836215097042e-01 6.71322342063399979e-03
            -3.13874021967429528e-01 7.78833599977760138e-02 6.71356749662900022e-03
            -1.78373469242165933e-01 3.09488622953186132e-01 6.59815299243199978e-03
            -3.11172817600997575e-01 2.31204046787292228e-01 6.48498667156699980e-03
            -3.55728141012735311e-01 1.54287585902669139e-01 6.48557078261800032e-03
            -2.65817507552782550e-01 3.07298132223529308e-01 6.40829273834399998e-03
            -3.99074900379234432e-01 7.71440253659819442e-02 6.40939593785600035e-03
            -2.20120814674428733e-01 3.81955879564629119e-01 6.25791916856600038e-03
            -3.95796045889001913e-01 2.28594665560714366e-01 6.18785584192700026e-03
            -3.51168313463117165e-01 3.04167837123617113e-01 6.14860221255199985e-03
            -4.39193707886702922e-01 1.52372062091011617e-01 6.15087493594800028e-03
            -3.05780089562288637e-01 3.78514019909734656e-01 6.03512873584500023e-03
            -4.80950033003492206e-01 7.60604965754970858e-02 6.03790924661300014e-03
            -2.60178509942715241e-01 4.51037370145901673e-01 5.85171462448200001e-03
            -4.33800899370147375e-01 3.00017106768889263e-01 5.82589813729699965e-03
            -4.77291155358440244e-01 2.25145924956303284e-01 5.82831101447100000e-03
            -3.89098699056349695e-01 3.73818363438207457e-01 5.74797153569299966e-03
            -5.19095316564420872e-01 1.49751503555659193e-01 5.75425064206199974e-03
            -3.43753026114304638e-01 4.45990525608388144e-01 5.59837142076200032e-03
            -5.58803060817919861e-01 7.43646278806359279e-02 5.60506991703199967e-03
            -5.13097584867880152e-01 2.94695606157365786e-01 5.44938025848299996e-03
            -2.98459298494306080e-01 5.15956124409567618e-01 5.38478976123000011e-03
            -4.69473381133585443e-01 3.67817055625677014e-01 5.40646093270800033e-03
            -5.55027087851630574e-01 2.20602731011358105e-01 5.41531091663099984e-03
            -4.24729746765085681e-01 4.39431618728079787e-01 5.28847197034399991e-03
            -5.94790845382857203e-01 1.46061538861841061e-01 5.30474416620400034e-03
            -3.79613963812988009e-01 5.08993291156477512e-01 5.10231770740100007e-03
            -6.31985512250962334e-01 7.15929571281656019e-02 5.11945714409100042e-03
            -5.46223593772422578e-01 3.60519209514614192e-01 5.02236053980400013e-03
            -5.88377081975176575e-01 2.88062981003293805e-01 5.02983271602400021e-03
            -5.02425319243467228e-01 4.31524971891165621e-01 4.93660000075799976e-03
            -6.28327928184265527e-01 2.14668436708806920e-01 4.96045900220599967e-03
            -3.35057290535509089e-01 5.75939377040328604e-01 4.86371774827700030e-03
            -4.57746728379097190e-01 5.00540596863971232e-01 4.77474165960099957e-03
            -6.65622755416226841e-01 1.40832181239163273e-01 4.81565372781499971e-03
            -4.13277746204846808e-01 5.67004988493295947e-01 4.55063054068700033e-03
            -6.99900763966257000e-01 6.71434415091130138e-02 4.59544679538400041e-03
            -6.18585417342566890e-01 3.51967814335717610e-01 4.60203789027800042e-03
            -5.76096850358223778e-01 4.22459701351574013e-01 4.55465787132300001e-03
            -6.58844026527364734e-01 2.80063336464448420e-01 4.57230938209900011e-03
            -5.32089064273337131e-01 4.91019535056152312e-01 4.42444979191600023e-03
            -6.96377353604080263e-01 2.07189008510516359e-01 4.47071236481500021e-03
            -3.70189207420812350e-01 6.30372024508871309e-01 4.30104406947099966e-03
            -4.87724673958910393e-01 5.57038589445215448e-01 4.21295726809199980e-03
            -7.30808444815164604e-01 1.33815109548702516e-01 4.29875887114499963e-03
            -4.44722101728094787e-01 6.19840530746552432e-01 3.95316724490200021e-03
            -7.61946088601781590e-01 6.07809748757690838e-02 4.05258060727699959e-03
            -6.45714179775940367e-01 4.11130951362094321e-01 4.13728997613100041e-03
            -6.86324315925145756e-01 3.41141040360261771e-01 4.13336894192800012e-03
            -6.02771908791406741e-01 4.79401544393785961e-01 4.05842545064200033e-03
            -7.24039654335587146e-01 2.69883581177304543e-01 4.06045140728300037e-03
            -5.58536399138914952e-01 5.45325380497324375e-01 3.88425009578400020e-03
            -7.58515223246489123e-01 1.97702843547853918e-01 3.92810303565400021e-03
            -4.03838070891964773e-01 6.78970938092292586e-01 3.72535039921600020e-03
            -5.15020511689508687e-01 6.08066281315069124e-01 3.61910327768300011e-03
            -7.89602549702961065e-01 1.25139315653742422e-01 3.73821698265100000e-03
            -7.12693622202636878e-01 3.93573769984556399e-01 3.70036558101400010e-03
            -6.71326807369703626e-01 4.62170664116544505e-01 3.68592864226299994e-03
            -7.50760360332662136e-01 3.23660380577337981e-01 3.62603685967500004e-03
            -4.74729082519349865e-01 6.66654629193252801e-01 3.35280890303199996e-03
            -8.17429042974247233e-01 5.33517142534850469e-02 3.48157001957200007e-03
            -6.27523209872855259e-01 5.28648564227507278e-01 3.56583693490600002e-03
            -7.85078302975277698e-01 2.53114368652765764e-01 3.48356058743999998e-03
            -5.83084062525879743e-01 5.92048346359227096e-01 3.32565976192399993e-03
            -8.15539754752635071e-01 1.82585169563481947e-01 3.29489471807600017e-03
            -4.35231873290285898e-01 7.21167964532798633e-01 3.16058812290699988e-03
            -5.41735333939928121e-01 6.51137890879997627e-01 3.01445684751700015e-03
            -8.42374315667918983e-01 1.12813681797584484e-01 3.06783191591500012e-03
            -7.34129402763544547e-01 4.43356345568128307e-01 3.32665344726499994e-03
            -7.74013762566669872e-01 3.72722040232869367e-01 3.30307358571299992e-03
            -6.90846005578866218e-01 5.11614282088506123e-01 3.25408345210300006e-03
            -8.09910131959323487e-01 3.00480722355480678e-01 3.19930874407000016e-03
            -6.45411678798878841e-01 5.76705684163720811e-01 3.06215445555199986e-03
            -8.65822197885722122e-01 4.43877397712505936e-02 2.79080351623500018e-03
            -5.04431927858103757e-01 7.05463301795430264e-01 2.87163083124200016e-03
            -8.41533323680639600e-01 2.27634031127901620e-01 3.02750821415299996e-03
            -6.01304872720919481e-01 6.37461136227477909e-01 2.72451847997900011e-03
            -8.68828887511436276e-01 1.55386505162838973e-01 2.78664812139600010e-03
            -4.62255601577678366e-01 7.56506327453456917e-01 2.54475514915600016e-03
            -5.65401184168216608e-01 6.92352297489326696e-01 2.57116546832600020e-03
            -8.91727947757210648e-01 8.47349039883326521e-02 2.47222339185300009e-03
            -7.86880322440087321e-01 4.28773721930038099e-01 2.87470048159199978e-03
            -7.45052433022597826e-01 4.98228536767768626e-01 2.86031186141600009e-03
            -8.24438230682060458e-01 3.57033085321203092e-01 2.81094637882399988e-03
            -6.99588452627412205e-01 5.64672011410554897e-01 2.74075746044699989e-03
            -8.57500182284304802e-01 2.83961100709827607e-01 2.68775126324999987e-03
            -6.52628757351211286e-01 6.27300169089228032e-01 2.46112952592700013e-03
            -5.22693939924904316e-01 7.44436983618613213e-01 2.36556594380100006e-03
            -8.85913956633483379e-01 2.10466987779419273e-01 2.49944003710999988e-03
            -9.11164729794803674e-01 1.90465260641985816e-02 2.14635104513900008e-03
            -9.09565255811758844e-01 1.37084816175052165e-01 2.25515678876800005e-03
            -6.16137141379996978e-01 6.84090054311462636e-01 2.29174836455199983e-03
            -7.86280242607006352e-01 4.93967293352012338e-01 2.26703879807000014e-03
            -8.23453559807724833e-01 4.29716955270073031e-01 2.11694199000600014e-03
            -7.43149243546308402e-01 5.58469839822081315e-01 2.27988297223499998e-03
            -8.54490910724784225e-01 3.68185642606376817e-01 1.92861219257499990e-03
            -9.29443881860226195e-01 6.73259409068611980e-02 1.97013878030600012e-03
            -4.94024813856773826e-01 7.90292608906129956e-01 2.04474818264299999e-03
            -6.95269296733224573e-01 6.20972966875368959e-01 2.10252607917200008e-03
            -8.81155694292178659e-01 3.07857307304890526e-01 1.87858335006200008e-03
            -5.72812245849653401e-01 7.37473310204335264e-01 2.07844803552899998e-03
            -9.05344834765734463e-01 2.43809170267323932e-01 1.87233680736699991e-03
            -9.26764657678185166e-01 1.75861926541168306e-01 1.78430481805599991e-03
            -6.58961249047239184e-01 6.77708522630763865e-01 1.91159890891400008e-03
            -9.51066580540543605e-01 1.71113756032346340e-02 1.77778576656199990e-03
            -9.45439216034880348e-01 1.08991758915244985e-01 1.61669914040900005e-03
            -5.41588437052337168e-01 7.84186807249426909e-01 1.75720749155900010e-03
            -8.15598421005140017e-01 4.97273610043488301e-01 1.65060830486800001e-03
            -8.49955624368077256e-01 4.36307958215732705e-01 1.64584059562500004e-03
            -6.15687068344501931e-01 7.30697376948973187e-01 1.68314707775099994e-03
            -7.76946752851826128e-01 5.56318823964717657e-01 1.69293087211900011e-03
            -7.31035722545664624e-01 6.15500473543519777e-01 1.65361412771100006e-03
            -8.78558461783060185e-01 3.78129169632169027e-01 1.13642016279599995e-03
            -8.91460719178329808e-01 3.49583372500583800e-01 8.56057631169000006e-04
            -9.13948855774347368e-01 2.91994816735847651e-01 1.58908904780400003e-03
            -9.37253248199679145e-01 2.20639451316991531e-01 1.51893311022200002e-03
            -6.96302472875299006e-01 6.69552490213461748e-01 1.46681544418000008e-03
            -9.56710953437757361e-01 1.51070487782215018e-01 1.33401257950899990e-03
            -9.66579174456633994e-01 6.37480423939577323e-02 1.42675368759600011e-03
            -5.07302053538506326e-01 8.25875548968226680e-01 1.43941250634100010e-03
            -5.83747243071622735e-01 7.76099692832157850e-01 1.37476644712400007e-03
            -6.53060942059506599e-01 7.21463082098862007e-01 1.26363484330699999e-03
            -7.62899010293302982e-01 6.06776086121726399e-01 1.23636098643800001e-03
            -8.46016719712465615e-01 4.85992489812186013e-01 1.25540140813499993e-03
            -8.79567831589894156e-01 4.23117472875050304e-01 1.17585700181700010e-03
            -8.08093044043369790e-01 5.47520365136915044e-01 1.27531648584499994e-03
            -9.02285147209507588e-01 3.74612389381783983e-01 6.45429343702000010e-04
            -9.19676521701747296e-01 3.31153037619921986e-01 1.03253300932900000e-03
            -9.43052041219566650e-01 2.62958905388836039e-01 1.15965694635099996e-03
            -7.28720974025778068e-01 6.57911808685515132e-01 9.87631064347000050e-04
            -9.63410492212333991e-01 1.91576851819840105e-01 1.02216970241799992e-03
            -9.76817316792356061e-01 1.07885987286374690e-01 1.06242630376100007e-03
            -9.82725591649050600e-01 2.02921281370708750e-02 1.07433276771499997e-03
            -5.49872640186208850e-01 8.15536361037160762e-01 1.04569291981000006e-03
            -6.22478381493084210e-01 7.63452996438860110e-01 9.70235686178999969e-04
            -6.87142649054556354e-01 7.07539418397016284e-01 8.65312708298999990e-04
            -7.89618984759697939e-01 5.95021825658519354e-01 6.78618929186000034e-04
            -8.53906319650844847e-01 5.00543120774045169e-01 7.12640554481999981e-04
            -8.86261194059197344e-01 4.41757313245703997e-01 7.79849561438999994e-04
            -9.16582090616369305e-01 3.75742411917608099e-01 7.99787468057999956e-04
            -8.23870975282177009e-01 5.50717857117026699e-01 6.10870396132000043e-04
            -9.43398028170163938e-01 3.03582759025566196e-01 7.87863412149000016e-04
            -7.58868822540552435e-01 6.39166663217348607e-01 5.71748160716000040e-04
            -9.65392221263422057e-01 2.29965952792616474e-01 6.86081209356000045e-04
            -9.81611350814131312e-01 1.48957699029185264e-01 6.82048477971000005e-04
            -9.90892510485991251e-01 6.35829952990793340e-02 6.89150458834000010e-04
            -5.16600332860018163e-01 8.48148655112635597e-01 6.79102356988000038e-04
            -5.89467986408988343e-01 7.99759698064401503e-01 6.42916727870000053e-04
            -6.58718901141280955e-01 7.44925448840322635e-01 5.76658274574999965e-04
            -7.19723703876285259e-01 6.86950754560743393e-01 5.06490665150999991e-04
            -7.93608183002200041e-01 6.05129776605727976e-01 3.65614865653999979e-04
            -8.73666676898286698e-01 4.82524028522872350e-01 3.75571429357999999e-04
            -9.08362548219818766e-01 4.13888096273589967e-01 3.58493698192999981e-04
            -9.37926167885857809e-01 3.41670719872997186e-01 3.55000291663000022e-04
            -8.35582875247513512e-01 5.46181951703290425e-01 3.28602265489999989e-04
            -9.62020925920554104e-01 2.67304821120992631e-01 3.13506391891999988e-04
            -9.80741919643887128e-01 1.88079143952160827e-01 2.95303559031000000e-04
            -9.93114045489439379e-01 1.04944848326447460e-01 2.94712962742999993e-04
            -9.98461979887509510e-01 2.00085591752200073e-02 2.92691680609000020e-04
            -5.54942944282430517e-01 8.30342647349446406e-01 2.82028007154999985e-04
            -6.24151947882066849e-01 7.79811637463085350e-01 2.57150644410000014e-04
            -7.49116662041972248e-01 6.61078691723686940e-01 1.98235288888000004e-04
            -6.89711283602235103e-01 7.22841030429075215e-01 2.09409578575000003e-04
            -9.11898994730129947e-02 1.11675418504194272e-17 7.18170208769999992e-03
            -1.36420069879017669e-01 -7.87365002628742511e-02 7.10267554837199985e-03
            -1.81626070716434190e-01 3.98786798957102612e-05 7.06329804647600010e-03
            -2.26118199399790720e-01 -7.82399590881715373e-02 6.94577127469500014e-03
            -1.80940984854166292e-01 -1.56577233696842233e-01 6.94583072076000035e-03
            -2.70552830951540868e-01 1.37231362572234959e-04 6.86785783981600038e-03
            -2.69527211420165946e-01 -1.55430528676271135e-01 6.75208334917300028e-03
            -3.13917219890731403e-01 -7.74929017913241480e-02 6.71322342063399979e-03
            -2.24385979273877489e-01 -2.32881196612900937e-01 6.71356749662900022e-03
            -3.57211744280805876e-01 2.68355751715129557e-04 6.59815299243199978e-03
            -3.55814986776059772e-01 -1.53881541615999345e-01 6.48498667156699980e-03
            -3.11481039386652947e-01 -2.30925814006707281e-01 6.48557078261800032e-03
            -3.99036742817477030e-01 -7.65556481996069460e-02 6.40829273834399998e-03
            -2.66346135906748716e-01 -3.07036989058170173e-01 6.40939593785600035e-03
            -4.40843902165012680e-01 3.47722372532787566e-04 6.25791916856600038e-03
            -3.95866810489687304e-01 -2.28472097676949953e-01 6.18785584192700026e-03
            -4.39001230694778422e-01 -1.52036761901387846e-01 6.14860221255199985e-03
            -3.51554930541187360e-01 -3.04166877166660965e-01 6.15087493594800028e-03
            -4.80692801711543327e-01 -7.55563155775555439e-02 6.03512873584500023e-03
            -3.06345338760585839e-01 -3.78484698244239914e-01 6.03790924661300014e-03
            -5.20699075573833436e-01 1.97485943777229799e-04 5.85171462448200001e-03
            -4.76722885716840017e-01 -2.25674045654639971e-01 5.82589813729699965e-03
            -4.33627668249923548e-01 -3.00773303063882835e-01 5.82831101447100000e-03
            -5.18285548666786444e-01 -1.50060176243171278e-01 5.74797153569299966e-03
            -3.89236264616327099e-01 -3.74673979352483921e-01 5.75425064206199974e-03
            -5.58115638081190690e-01 -7.47035904385693755e-02 5.59837142076200032e-03
            -3.43803187296567114e-01 -4.46755332440501307e-01 5.60506991703199967e-03
            -5.11762673749872610e-01 -2.97007740037343371e-01 5.44938025848299996e-03
            -5.96060760224002895e-01 -4.95272286968008002e-04 5.38478976123000011e-03
            -5.53275604683822975e-01 -2.22667346649420589e-01 5.40646093270800033e-03
            -4.68561113125876505e-01 -3.70366192362330482e-01 5.41531091663099984e-03
            -5.92923818427177673e-01 -1.48110941077455899e-01 5.28847197034399991e-03
            -4.23888425861630724e-01 -4.42073212609056243e-01 5.30474416620400034e-03
            -6.30608102403852677e-01 -7.42586907151155939e-02 5.10231770740100007e-03
            -3.77994075730522738e-01 -5.11519029868972241e-01 5.11945714409100042e-03
            -5.85330590878151646e-01 -2.92783903596042339e-01 5.02236053980400013e-03
            -5.43658400426314858e-01 -3.65518009493415175e-01 5.02983271602400021e-03
            -6.24924247646848841e-01 -2.19350604023766438e-01 4.93660000075799976e-03
            -5.00072283672651441e-01 -4.36813729360414904e-01 4.96045900220599967e-03
            -6.66306776824463154e-01 -2.19843680676998308e-03 4.86371774827700030e-03
            -6.62354236699173149e-01 -1.46149996843527835e-01 4.77474165960099957e-03
            -4.54775624331603012e-01 -5.06030124907866918e-01 4.81565372781499971e-03
            -6.97679597210121072e-01 -7.44065327855273489e-02 4.55063054068700033e-03
            -4.08098308027534884e-01 -5.72560120968358288e-01 4.59544679538400041e-03
            -6.14105777200499525e-01 -3.59726778661403179e-01 4.60203789027800042e-03
            -6.53909258624762124e-01 -2.87684656774637060e-01 4.55465787132300001e-03
            -5.71963977310523397e-01 -4.30543995872102347e-01 4.57230938209900011e-03
            -6.91279923249720207e-01 -2.15292879208524801e-01 4.42444979191600023e-03
            -5.27619621557057550e-01 -4.99485974586054371e-01 4.47071236481500021e-03
            -7.31012790770115495e-01 -5.40724557881479273e-03 4.30104406947099966e-03
            -7.26271906327262129e-01 -1.43862662978291395e-01 4.21295726809199980e-03
            -4.81291506686956250e-01 -5.65991123735779422e-01 4.29875887114499963e-03
            -7.59158696785791287e-01 -7.52203723476613517e-02 3.95316724490200021e-03
            -4.33610912610090438e-01 -6.29474181605447103e-01 4.05258060727699959e-03
            -6.78906938049608333e-01 -3.53639407588749177e-01 4.13728997613100041e-03
            -6.38598965188012291e-01 -4.23803772646021870e-01 4.13336894192800012e-03
            -7.16559870454215431e-01 -2.82315013504101986e-01 4.05842545064200033e-03
            -5.95745864531659008e-01 -4.92094943413269981e-01 4.06045140728300037e-03
            -7.51533832408555424e-01 -2.11044020343923083e-01 3.88425009578400020e-03
            -5.50473296536106327e-01 -5.58042030714757553e-01 3.92810303565400021e-03
            -7.89925116265259186e-01 -1.02485593615963145e-02 3.72535039921600020e-03
            -7.84111102648339164e-01 -1.41987705935640529e-01 3.61910327768300011e-03
            -5.03175101219821341e-01 -6.21246209108857861e-01 3.73821698265100000e-03
            -6.97191694171157672e-01 -4.20423896950354703e-01 3.70036558101400010e-03
            -7.35914939693704473e-01 -3.50300737365393466e-01 3.68592864226299994e-03
            -6.55678291944844949e-01 -4.88347353913775761e-01 3.62603685967500004e-03
            -8.14704385691526922e-01 -7.78001307804098291e-02 3.35280890303199996e-03
            -4.54918461366089544e-01 -6.81238459880157299e-01 3.48157001957200007e-03
            -7.71584691231618325e-01 -2.79126759100493005e-01 3.56583693490600002e-03
            -6.11742624803793600e-01 -5.53340570010183885e-01 3.48356058743999998e-03
            -8.04270939478598734e-01 -2.08941437509632399e-01 3.32565976192399993e-03
            -5.65893272572582040e-01 -6.14985560630171957e-01 3.29489471807600017e-03
            -8.42165714326061576e-01 -1.63378765396783344e-02 3.16058812290699988e-03
            -8.34769621838661657e-01 -1.43587615879625236e-01 3.01445684751700015e-03
            -5.18886672165121743e-01 -6.73110715965157480e-01 3.06783191591500012e-03
            -7.51022559572803727e-01 -4.14096539674263409e-01 3.32665344726499994e-03
            -7.09793636675365436e-01 -4.83954561145078233e-01 3.30307358571299992e-03
            -7.88493968017017322e-01 -3.42483049890051072e-01 3.25408345210300006e-03
            -6.65179004887006542e-01 -5.51162387881440785e-01 3.19930874407000016e-03
            -8.22147612392106764e-01 -2.70590067657131117e-01 3.06215445555199986e-03
            -4.71352009201336852e-01 -7.27630148643887420e-01 2.79080351623500018e-03
            -8.63165104721542620e-01 -8.41192131073620619e-02 2.87163083124200016e-03
            -6.17903515562940209e-01 -6.14972220874635878e-01 3.02750821415299996e-03
            -8.52709974258748393e-01 -2.02014727081945872e-01 2.72451847997900011e-03
            -5.68983104632018732e-01 -6.74734635545256589e-01 2.78664812139600010e-03
            -8.86281498487202057e-01 -2.20719302811991196e-02 2.54475514915600016e-03
            -8.82295270078386262e-01 -1.43475640074816252e-01 2.57116546832600020e-03
            -5.19246553319736681e-01 -7.29891604028140906e-01 2.47222339185300009e-03
            -7.64769096886661504e-01 -4.67071488006186919e-01 2.87470048159199978e-03
            -8.04004786242535796e-01 -3.96120065765089513e-01 2.86031186141600009e-03
            -7.21418837220728970e-01 -5.35467908961158146e-01 2.81094637882399988e-03
            -8.38814533001303042e-01 -3.23525366464307906e-01 2.74075746044699989e-03
            -6.74667618043454431e-01 -6.00636391253081103e-01 2.68775126324999987e-03
            -8.69572260905150829e-01 -2.51542998561805620e-01 2.46112952592700013e-03
            -9.06048309292831200e-01 -8.04477385698379455e-02 2.36556594380100006e-03
            -6.25226736391707338e-01 -6.61990498122072779e-01 2.49944003710999988e-03
            -4.72077140322839817e-01 -7.79568540002584798e-01 2.14635104513900008e-03
            -5.73501561186594455e-01 -7.19164209845148616e-01 2.25515678876800005e-03
            -9.00507936200001469e-01 -1.91545389494470447e-01 2.29174836455199983e-03
            -8.20928345984985874e-01 -4.33955017915452934e-01 2.26703879807000014e-03
            -7.83872579604646846e-01 -4.98273223995181813e-01 2.11694199000600014e-03
            -8.55223690306502915e-01 -3.64351203803251311e-01 2.27988297223499998e-03
            -7.46103575168212574e-01 -5.55918014687375783e-01 1.92861219257499990e-03
            -5.23027916089144340e-01 -7.71259042629548142e-01 1.97013878030600012e-03
            -9.31425882664175586e-01 -3.26917344467798354e-02 2.04474818264299999e-03
            -8.85413012744074468e-01 -2.91634390004629096e-01 2.10252607917200008e-03
            -7.07190096012797142e-01 -6.09174562293896327e-01 1.87858335006200008e-03
            -9.25076744174782695e-01 -1.27333301402449695e-01 2.07844803552899998e-03
            -6.63817352509975311e-01 -6.62147040958489197e-01 1.87233680736699991e-03
            -6.15683224782217375e-01 -7.14670773608312992e-01 1.78430481805599991e-03
            -9.16393421483082227e-01 -2.31822920469051591e-01 1.91159890891400008e-03
            -4.90352176236370563e-01 -8.15092131636892092e-01 1.77778576656199990e-03
            -5.67109240041191232e-01 -7.64278499362627928e-01 1.61669914040900005e-03
            -9.49919914916783292e-01 -7.69359412585199770e-02 1.75720749155900010e-03
            -8.38450789431827381e-01 -4.57692146855182946e-01 1.65060830486800001e-03
            -8.02831587872182673e-01 -5.17929183684352146e-01 1.64584059562500004e-03
            -9.40646025088715643e-01 -1.67851953493418055e-01 1.68314707775099994e-03
            -8.70259610582841736e-01 -3.94696213375152483e-01 1.69293087211900011e-03
            -8.98556907402872085e-01 -3.25345270026698274e-01 1.65361412771100006e-03
            -7.66748697704903392e-01 -5.71789361797825801e-01 1.13642016279599995e-03
            -7.48478440915308463e-01 -5.97235943034087668e-01 8.56057631169000006e-04
            -7.09849356953799249e-01 -6.45505518492381136e-01 1.58908904780400003e-03
            -6.59705994017414032e-01 -7.01365397061908147e-01 1.51893311022200002e-03
            -9.28000702129639143e-01 -2.68239385121203211e-01 1.46681544418000008e-03
            -6.09186356900383807e-01 -7.53000745864821242e-01 1.33401257950899990e-03
            -5.38497011383011981e-01 -8.05208098651456705e-01 1.42675368759600011e-03
            -9.68880232540156539e-01 -2.63986912722466993e-02 1.43941250634100010e-03
            -9.63995671397759657e-01 -1.17490095473076150e-01 1.37476644712400007e-03
            -9.51335828019985841e-01 -2.04835824993499144e-01 1.26363484330699999e-03
            -9.06933010136960926e-01 -3.57301880375143210e-01 1.23636098643800001e-03
            -8.43890202082035912e-01 -4.89675726391281374e-01 1.25540140813499993e-03
            -8.06214396089813712e-01 -5.50169350070916163e-01 1.17585700181700010e-03
            -8.78213067319585039e-01 -4.26068922194598076e-01 1.27531648584499994e-03
            -7.75566419381766625e-01 -5.94095664249923550e-01 6.45429343702000010e-04
            -7.46625203970109674e-01 -6.30886712247863080e-01 1.03253300932900000e-03
            -6.99255112827864012e-01 -6.85227572092496295e-01 1.15965694635099996e-03
            -9.34128826784312527e-01 -3.02134971434106281e-01 9.87631064347000050e-04
            -6.47615666559195557e-01 -7.38549534618431536e-01 1.02216970241799992e-03
            -5.81840664098543403e-01 -7.92005617555544772e-01 1.06242630376100007e-03
            -5.08936294288077851e-01 -8.40919263248634863e-01 1.07433276771499997e-03
            -9.81211526461203354e-01 -6.84354947286966164e-02 1.04569291981000006e-03
            -9.72408880257945474e-01 -1.57355593460202170e-01 9.70235686178999969e-04
            -9.56318435037960990e-01 -2.41313280906472999e-01 8.65312708298999990e-04
            -9.10113509206322058e-01 -3.86319187283115928e-01 6.78618929186000034e-04
            -8.60436218105287698e-01 -4.89233004882684730e-01 7.12640554481999981e-04
            -8.25703652607938210e-01 -5.46646051820743328e-01 7.79849561438999994e-04
            -7.83693519308070297e-01 -6.05912169168821846e-01 7.99787468057999956e-04
            -8.88871142222162347e-01 -4.38134265476513463e-01 6.10870396132000043e-04
            -7.34609395552192068e-01 -6.65215278762726125e-01 7.87863412149000016e-04
            -9.32968978868632948e-01 -3.37616346851428828e-01 5.71748160716000040e-04
            -6.81852467755609815e-01 -7.21071211833703152e-01 6.86081209356000045e-04
            -6.19806826855616277e-01 -7.75621516933604038e-01 6.82048477971000005e-04
            -5.50510744420704357e-01 -8.26346588851067265e-01 6.89150458834000010e-04
            -9.92818447943157878e-01 -2.33146843039549294e-02 6.79102356988000038e-04
            -9.87346208651238233e-01 -1.10614401915643437e-01 6.42916727870000053e-04
            -9.74483813191885040e-01 -1.98004577921158331e-01 5.76658274574999965e-04
            -9.54778656536635184e-01 -2.79823633982320130e-01 5.06490665150999991e-04
            -9.20861850628062695e-01 -3.84719958828251107e-01 3.65614865653999979e-04
            -8.54711405086357789e-01 -5.15355522372411334e-01 3.75571429357999999e-04
            -8.12618879806817684e-01 -5.79720994467935347e-01 3.58493698192999981e-04
            -7.64858607082260944e-01 -6.41432528326842721e-01 3.55000291663000022e-04
            -8.90798882887371524e-01 -4.50545021079944930e-01 3.28602265489999989e-04
            -7.12503228605111394e-01 -6.99482150258931457e-01 3.13506391891999988e-04
            -6.53252276406545151e-01 -7.55307844991842559e-01 2.95303559031000000e-04
            -5.87441927391727958e-01 -8.07589568085765430e-01 2.94712962742999993e-04
            -5.16558910482618927e-01 -8.54689159707880730e-01 2.92691680609000020e-04
            -9.96569298591459374e-01 -6.54233637247940425e-02 2.82028007154999985e-04
            -9.87412662150806142e-01 -1.50625623955868204e-01 2.57150644410000014e-04
            -9.47069271954280367e-01 -3.18214713864706900e-01 1.98235288888000004e-04
            -9.70854337050417127e-01 -2.35886977661771663e-01 2.09409578575000003e-04
            -4.55949497365065390e-02 -7.89727695121784368e-02 7.18170208769999992e-03
            -2.22255067795913393e-05 -1.57511496232714737e-01 7.10267554837199985e-03
            -9.08475713080761499e-02 -1.57272851890033094e-01 7.06329804647600010e-03
            -4.53013075384837219e-02 -2.34944084482299725e-01 6.94577127469500014e-03
            4.51293696086749671e-02 -2.34988106317904510e-01 6.94583072076000035e-03
            -1.35395261321953897e-01 -2.34237008988545048e-01 6.86785783981600038e-03
            -1.56819352786733140e-04 -3.11132676439178524e-01 6.75208334917300028e-03
            -8.98477883811063249e-02 -3.10606738006421190e-01 6.71322342063399979e-03
            8.94880426935519141e-02 -3.10764556610676979e-01 6.71356749662900022e-03
            -1.78838275038639999e-01 -3.09220267201470944e-01 6.59815299243199978e-03
            -4.46421691750624391e-02 -3.85085588403291490e-01 6.48498667156699980e-03
            4.42471016260824399e-02 -3.85213399909376364e-01 6.48557078261800032e-03
            -1.33219235264694785e-01 -3.83853780423136060e-01 6.40829273834399998e-03
            1.32728764472485633e-01 -3.84181014424152145e-01 6.40939593785600035e-03
            -2.20723087490584058e-01 -3.81608157192096176e-01 6.25791916856600038e-03
            -7.07646006855100831e-05 -4.57066763237664264e-01 6.18785584192700026e-03
            -8.78329172316613810e-02 -4.56204599025004820e-01 6.14860221255199985e-03
            8.76387773455154645e-02 -4.56538939257672527e-01 6.15087493594800028e-03
            -1.74912712149254607e-01 -4.54070335487290200e-01 6.03512873584500023e-03
            1.74604694242906228e-01 -4.54545194819736986e-01 6.03790924661300014e-03
            -2.60520565631118528e-01 -4.50839884202124230e-01 5.85171462448200001e-03
            -4.29219863466927531e-02 -5.25691152423529151e-01 5.82589813729699965e-03
            4.36634871085165438e-02 -5.25919228020186091e-01 5.82831101447100000e-03
            -1.29186849610437193e-01 -5.23878539681378541e-01 5.74797153569299966e-03
            1.29859051948093551e-01 -5.24425482908143281e-01 5.75425064206199974e-03
            -2.14362611966886163e-01 -5.20694116046957367e-01 5.59837142076200032e-03
            2.14999873521352525e-01 -5.21119960321137277e-01 5.60506991703199967e-03
            1.33491111800730883e-03 -5.91703346194709101e-01 5.44938025848299996e-03
            -2.97601461729697148e-01 -5.16451396696535392e-01 5.38478976123000011e-03
            -8.38022235502376422e-02 -5.90484402275097575e-01 5.40646093270800033e-03
            8.64659747257535971e-02 -5.90968923373688670e-01 5.41531091663099984e-03
            -1.68194071662092048e-01 -5.87542559805535491e-01 5.28847197034399991e-03
            1.70902419521226340e-01 -5.88134751470896999e-01 5.30474416620400034e-03
            -2.50994138590864557e-01 -5.83251981871592995e-01 5.10231770740100007e-03
            2.53991436520439429e-01 -5.83111986997137843e-01 5.11945714409100042e-03
            -3.91069971057293186e-02 -6.53303113110656586e-01 5.02236053980400013e-03
            4.47186815488615150e-02 -6.53580990496708925e-01 5.02983271602400021e-03
            -1.22498928403381807e-01 -6.50875575914931948e-01 4.93660000075799976e-03
            1.28255644511613559e-01 -6.51482166069221824e-01 4.96045900220599967e-03
            -3.31249486288954509e-01 -5.78137813847098281e-01 4.86371774827700030e-03
            -2.04607508320076154e-01 -6.46690593707498929e-01 4.77474165960099957e-03
            2.10847131084623607e-01 -6.46862306147030108e-01 4.81565372781499971e-03
            -2.84401851005274653e-01 -6.41411521278823060e-01 4.55063054068700033e-03
            2.91802455938721894e-01 -6.39703562477471399e-01 4.59544679538400041e-03
            4.47964014206707947e-03 -7.11694592997120679e-01 4.60203789027800042e-03
            -7.78124082665388878e-02 -7.10144358126211017e-01 4.55465787132300001e-03
            8.68800492168411709e-02 -7.10607332336550712e-01 4.57230938209900011e-03
            -1.59190858976383270e-01 -7.06312414264677058e-01 4.42444979191600023e-03
            1.68757732047022879e-01 -7.06674983096570619e-01 4.47071236481500021e-03
            -3.60823583349303256e-01 -6.35779270087685866e-01 4.30104406947099966e-03
            -2.38547232368351875e-01 -7.00901252423506760e-01 4.21295726809199980e-03
            2.49516938128208188e-01 -6.99806233284481882e-01 4.29875887114499963e-03
            -3.14436595057696278e-01 -6.95060903094213756e-01 3.95316724490200021e-03
            3.28335175991690931e-01 -6.90255156481216270e-01 4.05258060727699959e-03
            -3.31927582736681881e-02 -7.64770358950843554e-01 4.13728997613100041e-03
            4.77253507371332153e-02 -7.64944813006283919e-01 4.13336894192800012e-03
            -1.13787961662809120e-01 -7.61716557897887836e-01 4.05842545064200033e-03
            1.28293789803927527e-01 -7.61978524590574469e-01 4.06045140728300037e-03
            -1.92997433269640362e-01 -7.56369400841247375e-01 3.88425009578400020e-03
            2.08041926710382491e-01 -7.55744874262611388e-01 3.92810303565400021e-03
            -3.86087045373294580e-01 -6.89219497453888774e-01 3.72535039921600020e-03
            -2.69090590958830589e-01 -7.50053987250709486e-01 3.61910327768300011e-03
            2.86427448483139446e-01 -7.46385524762600561e-01 3.73821698265100000e-03
            1.55019280314790345e-02 -8.13997666934911046e-01 3.70036558101400010e-03
            -6.45881323240013877e-02 -8.12471401481937971e-01 3.68592864226299994e-03
            9.50820683878170619e-02 -8.12007734491113298e-01 3.62603685967500004e-03
            -3.39975303172177445e-01 -7.44454759973662283e-01 3.35280890303199996e-03
            3.62510581608157356e-01 -7.34590174133642138e-01 3.48157001957200007e-03
            -1.44061481358763260e-01 -8.07775323328000061e-01 3.56583693490600002e-03
            1.73335678171484292e-01 -8.06454938662949594e-01 3.48356058743999998e-03
            -2.21186876952718880e-01 -8.00989783868859551e-01 3.32565976192399993e-03
            2.49646482180053059e-01 -7.97570730193653765e-01 3.29489471807600017e-03
            -4.06933841035776178e-01 -7.37505841072476520e-01 3.16058812290699988e-03
            -2.93034287898733814e-01 -7.94725506759622724e-01 3.01445684751700015e-03
            3.23487643502797018e-01 -7.85924397762742033e-01 3.06783191591500012e-03
            -1.68931568092593708e-02 -8.57452885242391716e-01 3.32665344726499994e-03
            6.42201258913042416e-02 -8.56676601377947544e-01 3.30307358571299992e-03
            -9.76479624381513811e-02 -8.54097331978557195e-01 3.25408345210300006e-03
            1.44731127072316640e-01 -8.51643110236921519e-01 3.19930874407000016e-03
            -1.76735933593228089e-01 -8.47295751820851817e-01 3.06215445555199986e-03
            3.94470188684385048e-01 -7.72017888415138076e-01 2.79080351623500018e-03
            -3.58733176863439474e-01 -7.89582514902792076e-01 2.87163083124200016e-03
            2.23629808117699141e-01 -8.42606252002537470e-01 3.02750821415299996e-03
            -2.51405101537829467e-01 -8.39475863309423698e-01 2.72451847997900011e-03
            2.99845782879417211e-01 -8.30121140708095950e-01 2.78664812139600010e-03
            -4.24025896909524191e-01 -7.78578257734655721e-01 2.54475514915600016e-03
            -3.16894085910169543e-01 -8.35827937564142975e-01 2.57116546832600020e-03
            3.72481394437473745e-01 -8.14626508016473627e-01 2.47222339185300009e-03
            2.21112255534255675e-02 -8.95845209936224962e-01 2.87470048159199978e-03
            -5.89523532199381708e-02 -8.94348602532858084e-01 2.86031186141600009e-03
            1.03019393461331196e-01 -8.92500994282361182e-01 2.81094637882399988e-03
            -1.39226080373890670e-01 -8.88197377874862748e-01 2.74075746044699989e-03
            1.82832564240850093e-01 -8.84597491962908600e-01 2.68775126324999987e-03
            -2.16943503553939515e-01 -8.78843167651033208e-01 2.46112952592700013e-03
            -3.83354369367927328e-01 -8.24884722188450770e-01 2.36556594380100006e-03
            2.60687220241775708e-01 -8.72457485901491525e-01 2.49944003710999988e-03
            4.39087589471963580e-01 -7.98615066066783008e-01 2.14635104513900008e-03
            3.36063694625164111e-01 -8.56249026020200810e-01 2.25515678876800005e-03
            -2.84370794820004658e-01 -8.75635443805932834e-01 2.29174836455199983e-03
            -3.46481033779793973e-02 -9.27922311267465161e-01 2.26703879807000014e-03
            3.95809802030780006e-02 -9.27990179265254733e-01 2.11694199000600014e-03
            -1.12074446760194721e-01 -9.22821043625332460e-01 2.27988297223499998e-03
            1.08387335556571013e-01 -9.24103657293752545e-01 1.92861219257499990e-03
            4.06415965771081578e-01 -8.38584983536408979e-01 1.97013878030600012e-03
            -4.37401068807401538e-01 -8.22984343352909820e-01 2.04474818264299999e-03
            -1.90143716010849811e-01 -9.12607356879997944e-01 2.10252607917200008e-03
            1.73965598279380934e-01 -9.17031869598786797e-01 1.87858335006200008e-03
            -3.52264498325129904e-01 -8.64806611606784625e-01 2.07844803552899998e-03
            2.41527482255759263e-01 -9.05956211225812935e-01 1.87233680736699991e-03
            3.11081432895968180e-01 -8.90532700149481382e-01 1.78430481805599991e-03
            -2.57432172435843154e-01 -9.09531443099815151e-01 1.91159890891400008e-03
            4.60714404304172820e-01 -8.32203507240127260e-01 1.77778576656199990e-03
            3.78329975993688727e-01 -8.73270258277872857e-01 1.61669914040900005e-03
            -4.08331477864446291e-01 -8.61122748507946678e-01 1.75720749155900010e-03
            -2.28523684266872008e-02 -9.54965756898671136e-01 1.65060830486800001e-03
            4.71240364958940908e-02 -9.54237141900085239e-01 1.64584059562500004e-03
            -3.24958956744213545e-01 -8.98549330442391270e-01 1.68314707775099994e-03
            -9.33128577310162188e-02 -9.51015037339870029e-01 1.69293087211900011e-03
            -1.67521184857207739e-01 -9.40845743570217774e-01 1.65361412771100006e-03
            1.11809764078156626e-01 -9.49918531429994495e-01 1.13642016279599995e-03
            1.42982278263021234e-01 -9.46819315534671024e-01 8.56057631169000006e-04
            2.04099498820547870e-01 -9.37500335228228732e-01 1.58908904780400003e-03
            2.77547254182264780e-01 -9.22004848378899622e-01 1.51893311022200002e-03
            -2.31698229254340748e-01 -9.37791875334664793e-01 1.46681544418000008e-03
            3.47524596537373109e-01 -9.04071233647036787e-01 1.33401257950899990e-03
            4.28082163073621624e-01 -8.68956141045414965e-01 1.42675368759600011e-03
            -4.61578179001650379e-01 -8.52274240240473158e-01 1.43941250634100010e-03
            -3.80248428326137533e-01 -8.93589788305233612e-01 1.37476644712400007e-03
            -2.98274885960479519e-01 -9.26298907092360957e-01 1.26363484330699999e-03
            -1.44033999843658611e-01 -9.64077966496869498e-01 1.23636098643800001e-03
            2.12651763042910266e-03 -9.75668216203467331e-01 1.25540140813499993e-03
            7.33534355000797222e-02 -9.73286822945966357e-01 1.17585700181700010e-03
            -7.01200232762159148e-02 -9.73589287331512954e-01 1.27531648584499994e-03
            1.26718727827740296e-01 -9.68708053631707644e-01 6.45429343702000010e-04
            1.73051317731637511e-01 -9.62039749867784622e-01 1.03253300932900000e-03
            2.43796928391702333e-01 -9.48186477481332335e-01 1.15965694635099996e-03
            -2.05407852758534876e-01 -9.60046780119621412e-01 9.87631064347000050e-04
            3.15794825653138600e-01 -9.30126386438271391e-01 1.02216970241799992e-03
            3.94976652693812380e-01 -8.99891604841919546e-01 1.06242630376100007e-03
            4.73789297360972361e-01 -8.61211391385706238e-01 1.07433276771499997e-03
            -4.31338886274995059e-01 -8.83971855765857017e-01 1.04569291981000006e-03
            -3.49930498764861486e-01 -9.20808589899062113e-01 9.70235686178999969e-04
            -2.69175785983404470e-01 -9.48852699303489144e-01 8.65312708298999990e-04
            -1.20494524446624757e-01 -9.81341012941635449e-01 6.78618929186000034e-04
            -6.52989845444292352e-03 -9.89776125656729455e-01 7.12640554481999981e-04
            6.05575414512589394e-02 -9.88403365066447215e-01 7.79849561438999994e-04
            1.32888571308298592e-01 -9.81654581086430333e-01 7.99787468057999956e-04
            -6.50001669399855181e-02 -9.88852122593540050e-01 6.10870396132000043e-04
            2.08788632617971454e-01 -9.68798037788292765e-01 7.87863412149000016e-04
            -1.74100156328081068e-01 -9.76783010068777657e-01 5.71748160716000040e-04
            2.83539753507812353e-01 -9.51037164626319376e-01 6.86081209356000045e-04
            3.61804523958514812e-01 -9.24579215962788914e-01 6.82048477971000005e-04
            4.40381766065286562e-01 -8.89929584150146225e-01 6.89150458834000010e-04
            -4.76218115083139548e-01 -8.71463339416590488e-01 6.79102356988000038e-04
            -3.97878222242250501e-01 -9.10374099980044593e-01 6.42916727870000053e-04
            -3.15764912050604307e-01 -9.42930026761480855e-01 5.76658274574999965e-04
            -2.35054952660350591e-01 -9.66774388543063301e-01 5.06490665150999991e-04
            -1.27253667625862904e-01 -9.89849735433978917e-01 3.65614865653999979e-04
            1.89552718119286104e-02 -9.97879550895283574e-01 3.75571429357999999e-04
            9.57436684130008042e-02 -9.93609090741525258e-01 3.58493698192999981e-04
            1.73067560803596032e-01 -9.83103248199839852e-01 3.55000291663000022e-04
            -5.52160076398579350e-02 -9.96726972783235299e-01 3.28602265489999989e-04
            2.49517697315442488e-01 -9.66786971379923643e-01 3.13506391891999988e-04
            3.27489643237341310e-01 -9.43386988944003413e-01 2.95303559031000000e-04
            4.05672118097711198e-01 -9.12534416412213001e-01 2.94712962742999993e-04
            4.81903069404890194e-01 -8.74697718883100439e-01 2.92691680609000020e-04
            -4.41626354309029023e-01 -8.95766011074240254e-01 2.82028007154999985e-04
            -3.63260714268739626e-01 -9.30437261418953443e-01 2.57150644410000014e-04
            -1.97952609912308120e-01 -9.79293405588393395e-01 1.98235288888000004e-04
            -2.81143053448182578e-01 -9.58728008090846573e-01 2.09409578575000003e-04
            4.55949497365064349e-02 -7.89727695121784784e-02 7.18170208769999992e-03
            1.36397844372238036e-01 -7.87749959698405139e-02 7.10267554837199985e-03
            9.07784994083579849e-02 -1.57312730569928810e-01 7.06329804647600010e-03
            1.80816891861306978e-01 -1.56704125394128313e-01 6.94577127469500014e-03
            2.26070354462841294e-01 -7.84108726210624024e-02 6.94583072076000035e-03
            1.35157569629586860e-01 -2.34374240351117286e-01 6.86785783981600038e-03
            2.69370392067379394e-01 -1.55702147762907472e-01 6.75208334917300028e-03
            2.24069431509625022e-01 -2.33113836215097042e-01 6.71322342063399979e-03
            3.13874021967429473e-01 -7.78833599977761942e-02 6.71356749662900022e-03
            1.78373469242165766e-01 -3.09488622953186243e-01 6.59815299243199978e-03
            3.11172817600997353e-01 -2.31204046787292533e-01 6.48498667156699980e-03
            3.55728141012735144e-01 -1.54287585902669500e-01 6.48557078261800032e-03
            2.65817507552782217e-01 -3.07298132223529530e-01 6.40829273834399998e-03
            3.99074900379234432e-01 -7.71440253659821662e-02 6.40939593785600035e-03
            2.20120814674428483e-01 -3.81955879564629175e-01 6.25791916856600038e-03
            3.95796045889001802e-01 -2.28594665560714561e-01 6.18785584192700026e-03
            3.51168313463116943e-01 -3.04167837123617280e-01 6.14860221255199985e-03
            4.39193707886702867e-01 -1.52372062091011867e-01 6.15087493594800028e-03
            3.05780089562288582e-01 -3.78514019909734711e-01 6.03512873584500023e-03
            4.80950033003492150e-01 -7.60604965754973494e-02 6.03790924661300014e-03
            2.60178509942714797e-01 -4.51037370145902006e-01 5.85171462448200001e-03
            4.33800899370147208e-01 -3.00017106768889485e-01 5.82589813729699965e-03
            4.77291155358440133e-01 -2.25145924956303561e-01 5.82831101447100000e-03
            3.89098699056349251e-01 -3.73818363438207846e-01 5.74797153569299966e-03
            5.19095316564420761e-01 -1.49751503555659721e-01 5.75425064206199974e-03
            3.43753026114304416e-01 -4.45990525608388311e-01 5.59837142076200032e-03
            5.58803060817919750e-01 -7.43646278806362471e-02 5.60506991703199967e-03
            5.13097584867879930e-01 -2.94695606157366119e-01 5.44938025848299996e-03
            2.98459298494305580e-01 -5.15956124409567951e-01 5.38478976123000011e-03
            4.69473381133585277e-01 -3.67817055625677292e-01 5.40646093270800033e-03
            5.55027087851630352e-01 -2.20602731011358660e-01 5.41531091663099984e-03
            4.24729746765085403e-01 -4.39431618728080009e-01 5.28847197034399991e-03
            5.94790845382857092e-01 -1.46061538861841117e-01 5.30474416620400034e-03
            3.79613963812987953e-01 -5.08993291156477512e-01 5.10231770740100007e-03
            6.31985512250962334e-01 -7.15929571281659488e-02 5.11945714409100042e-03
            5.46223593772422356e-01 -3.60519209514614580e-01 5.02236053980400013e-03
            5.88377081975176353e-01 -2.88062981003294138e-01 5.02983271602400021e-03
            5.02425319243467006e-01 -4.31524971891165898e-01 4.93660000075799976e-03
            6.28327928184265305e-01 -2.14668436708807558e-01 4.96045900220599967e-03
            3.35057290535508479e-01 -5.75939377040328937e-01 4.86371774827700030e-03
            4.57746728379096912e-01 -5.00540596863971454e-01 4.77474165960099957e-03
            6.65622755416226730e-01 -1.40832181239163662e-01 4.81565372781499971e-03
            4.13277746204846252e-01 -5.67004988493296391e-01 4.55063054068700033e-03
            6.99900763966257000e-01 -6.71434415091134162e-02 4.59544679538400041e-03
            6.18585417342566668e-01 -3.51967814335717943e-01 4.60203789027800042e-03
            5.76096850358223223e-01 -4.22459701351574624e-01 4.55465787132300001e-03
            6.58844026527364623e-01 -2.80063336464448809e-01 4.57230938209900011e-03
            5.32089064273336909e-01 -4.91019535056152645e-01 4.42444979191600023e-03
            6.96377353604080152e-01 -2.07189008510517053e-01 4.47071236481500021e-03
            3.70189207420812016e-01 -6.30372024508871531e-01 4.30104406947099966e-03
            4.87724673958910115e-01 -5.57038589445215782e-01 4.21295726809199980e-03
            7.30808444815164604e-01 -1.33815109548702904e-01 4.29875887114499963e-03
            4.44722101728094676e-01 -6.19840530746552543e-01 3.95316724490200021e-03
            7.61946088601781590e-01 -6.07809748757695070e-02 4.05258060727699959e-03
            6.45714179775940034e-01 -4.11130951362094710e-01 4.13728997613100041e-03
            6.86324315925145423e-01 -3.41141040360262493e-01 4.13336894192800012e-03
            6.02771908791406408e-01 -4.79401544393786572e-01 4.05842545064200033e-03
            7.24039654335586813e-01 -2.69883581177305265e-01 4.06045140728300037e-03
            5.58536399138914841e-01 -5.45325380497324486e-01 3.88425009578400020e-03
            7.58515223246489012e-01 -1.97702843547854362e-01 3.92810303565400021e-03
            4.03838070891964329e-01 -6.78970938092292808e-01 3.72535039921600020e-03
            5.15020511689508353e-01 -6.08066281315069457e-01 3.61910327768300011e-03
            7.89602549702960954e-01 -1.25139315653743227e-01 3.73821698265100000e-03
            7.12693622202636767e-01 -3.93573769984556787e-01 3.70036558101400010e-03
            6.71326807369703182e-01 -4.62170664116545227e-01 3.68592864226299994e-03
            7.50760360332662136e-01 -3.23660380577338036e-01 3.62603685967500004e-03
            4.74729082519349255e-01 -6.66654629193253245e-01 3.35280890303199996e-03
            8.17429042974247122e-01 -5.33517142534851510e-02 3.48157001957200007e-03
            6.27523209872855037e-01 -5.28648564227507611e-01 3.56583693490600002e-03
            7.85078302975277698e-01 -2.53114368652765875e-01 3.48356058743999998e-03
            5.83084062525879743e-01 -5.92048346359227318e-01 3.32565976192399993e-03
            8.15539754752635071e-01 -1.82585169563482058e-01 3.29489471807600017e-03
            4.35231873290285287e-01 -7.21167964532799077e-01 3.16058812290699988e-03
            5.41735333939927788e-01 -6.51137890879997960e-01 3.01445684751700015e-03
            8.42374315667918871e-01 -1.12813681797584955e-01 3.06783191591500012e-03
            7.34129402763544325e-01 -4.43356345568128751e-01 3.32665344726499994e-03
            7.74013762566669650e-01 -3.72722040232869867e-01 3.30307358571299992e-03
            6.90846005578865885e-01 -5.11614282088506678e-01 3.25408345210300006e-03
            8.09910131959323265e-01 -3.00480722355481067e-01 3.19930874407000016e-03
            6.45411678798878508e-01 -5.76705684163721144e-01 3.06215445555199986e-03
            8.65822197885722122e-01 -4.43877397712510932e-02 2.79080351623500018e-03
            5.04431927858102980e-01 -7.05463301795430819e-01 2.87163083124200016e-03
            8.41533323680639489e-01 -2.27634031127902092e-01 3.02750821415299996e-03
            6.01304872720918926e-01 -6.37461136227478686e-01 2.72451847997900011e-03
            8.68828887511436165e-01 -1.55386505162839861e-01 2.78664812139600010e-03
            4.62255601577677588e-01 -7.56506327453457472e-01 2.54475514915600016e-03
            5.65401184168216497e-01 -6.92352297489326807e-01 2.57116546832600020e-03
            8.91727947757210648e-01 -8.47349039883331517e-02 2.47222339185300009e-03
            7.86880322440087099e-01 -4.28773721930038543e-01 2.87470048159199978e-03
            7.45052433022597493e-01 -4.98228536767769126e-01 2.86031186141600009e-03
            8.24438230682060236e-01 -3.57033085321203536e-01 2.81094637882399988e-03
            6.99588452627412205e-01 -5.64672011410555008e-01 2.74075746044699989e-03
            8.57500182284304580e-01 -2.83961100709828052e-01 2.68775126324999987e-03
            6.52628757351211175e-01 -6.27300169089228032e-01 2.46112952592700013e-03
            5.22693939924903650e-01 -7.44436983618613657e-01 2.36556594380100006e-03
            8.85913956633483268e-01 -2.10466987779419384e-01 2.49944003710999988e-03
            9.11164729794803674e-01 -1.90465260641986892e-02 2.14635104513900008e-03
            9.09565255811758733e-01 -1.37084816175052693e-01 2.25515678876800005e-03
            6.16137141379996534e-01 -6.84090054311462969e-01 2.29174836455199983e-03
            7.86280242607006241e-01 -4.93967293352012449e-01 2.26703879807000014e-03
            8.23453559807724278e-01 -4.29716955270073864e-01 2.11694199000600014e-03
            7.43149243546308069e-01 -5.58469839822081648e-01 2.27988297223499998e-03
            8.54490910724783892e-01 -3.68185642606377705e-01 1.92861219257499990e-03
            9.29443881860226195e-01 -6.73259409068612952e-02 1.97013878030600012e-03
            4.94024813856773715e-01 -7.90292608906129956e-01 2.04474818264299999e-03
            6.95269296733224462e-01 -6.20972966875369070e-01 2.10252607917200008e-03
            8.81155694292178437e-01 -3.07857307304891414e-01 1.87858335006200008e-03
            5.72812245849652624e-01 -7.37473310204335819e-01 2.07844803552899998e-03
            9.05344834765734241e-01 -2.43809170267324821e-01 1.87233680736699991e-03
            9.26764657678184944e-01 -1.75861926541169250e-01 1.78430481805599991e-03
            6.58961249047238851e-01 -6.77708522630764088e-01 1.91159890891400008e-03
            9.51066580540543605e-01 -1.71113756032355950e-02 1.77778576656199990e-03
            9.45439216034880237e-01 -1.08991758915245526e-01 1.61669914040900005e-03
            5.41588437052336613e-01 -7.84186807249427242e-01 1.75720749155900010e-03
            8.15598421005140017e-01 -4.97273610043488412e-01 1.65060830486800001e-03
            8.49955624368076812e-01 -4.36307958215733538e-01 1.64584059562500004e-03
            6.15687068344501931e-01 -7.30697376948973298e-01 1.68314707775099994e-03
            7.76946752851825684e-01 -5.56318823964718434e-01 1.69293087211900011e-03
            7.31035722545664179e-01 -6.15500473543520221e-01 1.65361412771100006e-03
            8.78558461783060074e-01 -3.78129169632169138e-01 1.13642016279599995e-03
            8.91460719178329808e-01 -3.49583372500583911e-01 8.56057631169000006e-04
            9.13948855774347257e-01 -2.91994816735848151e-01 1.58908904780400003e-03
            9.37253248199679034e-01 -2.20639451316992086e-01 1.51893311022200002e-03
            6.96302472875298450e-01 -6.69552490213462415e-01 1.46681544418000008e-03
            9.56710953437757139e-01 -1.51070487782215990e-01 1.33401257950899990e-03
            9.66579174456633883e-01 -6.37480423939587176e-02 1.42675368759600011e-03
            5.07302053538505882e-01 -8.25875548968226902e-01 1.43941250634100010e-03
            5.83747243071621957e-01 -7.76099692832158405e-01 1.37476644712400007e-03
            6.53060942059506155e-01 -7.21463082098862341e-01 1.26363484330699999e-03
            7.62899010293302426e-01 -6.06776086121727176e-01 1.23636098643800001e-03
            8.46016719712465615e-01 -4.85992489812186124e-01 1.25540140813499993e-03
            8.79567831589893712e-01 -4.23117472875051193e-01 1.17585700181700010e-03
            8.08093044043369679e-01 -5.47520365136915155e-01 1.27531648584499994e-03
            9.02285147209507143e-01 -3.74612389381784927e-01 6.45429343702000010e-04
            9.19676521701747296e-01 -3.31153037619922097e-01 1.03253300932900000e-03
            9.43052041219566539e-01 -2.62958905388836595e-01 1.15965694635099996e-03
            7.28720974025777624e-01 -6.57911808685515576e-01 9.87631064347000050e-04
            9.63410492212333991e-01 -1.91576851819840216e-01 1.02216970241799992e-03
            9.76817316792355950e-01 -1.07885987286375232e-01 1.06242630376100007e-03
            9.82725591649050489e-01 -2.02921281370718673e-02 1.07433276771499997e-03
            5.49872640186208073e-01 -8.15536361037161317e-01 1.04569291981000006e-03
            6.22478381493083766e-01 -7.63452996438860443e-01 9.70235686178999969e-04
            6.87142649054556243e-01 -7.07539418397016395e-01 8.65312708298999990e-04
            7.89618984759697384e-01 -5.95021825658520132e-01 6.78618929186000034e-04
            8.53906319650844736e-01 -5.00543120774045280e-01 7.12640554481999981e-04
            8.86261194059197122e-01 -4.41757313245704553e-01 7.79849561438999994e-04
            9.16582090616368861e-01 -3.75742411917608987e-01 7.99787468057999956e-04
            8.23870975282176676e-01 -5.50717857117027254e-01 6.10870396132000043e-04
            9.43398028170163605e-01 -3.03582759025567139e-01 7.87863412149000016e-04
            7.58868822540551768e-01 -6.39166663217349273e-01 5.71748160716000040e-04
            9.65392221263421946e-01 -2.29965952792616612e-01 6.86081209356000045e-04
            9.81611350814131312e-01 -1.48957699029185403e-01 6.82048477971000005e-04
            9.90892510485991251e-01 -6.35829952990794589e-02 6.89150458834000010e-04
            5.16600332860018052e-01 -8.48148655112635597e-01 6.79102356988000038e-04
            5.89467986408987454e-01 -7.99759698064402169e-01 6.42916727870000053e-04
            6.58718901141280511e-01 -7.44925448840323079e-01 5.76658274574999965e-04
            7.19723703876285259e-01 -6.86950754560743504e-01 5.06490665150999991e-04
            7.93608183002199707e-01 -6.05129776605728420e-01 3.65614865653999979e-04
            8.73666676898286365e-01 -4.82524028522872850e-01 3.75571429357999999e-04
            9.08362548219818544e-01 -4.13888096273590467e-01 3.58493698192999981e-04
            9.37926167885857476e-01 -3.41670719872998130e-01 3.55000291663000022e-04
            8.35582875247513401e-01 -5.46181951703290536e-01 3.28602265489999989e-04
            9.62020925920554104e-01 -2.67304821120992742e-01 3.13506391891999988e-04
            9.80741919643886906e-01 -1.88079143952161798e-01 2.95303559031000000e-04
            9.93114045489439379e-01 -1.04944848326448029e-01 2.94712962742999993e-04
            9.98461979887509510e-01 -2.00085591752201288e-02 2.92691680609000020e-04
            5.54942944282430073e-01 -8.30342647349446739e-01 2.82028007154999985e-04
            6.24151947882066405e-01 -7.79811637463085683e-01 2.57150644410000014e-04
            7.49116662041972137e-01 -6.61078691723687051e-01 1.98235288888000004e-04
            6.89711283602234992e-01 -7.22841030429075326e-01 2.09409578575000003e-04
            ];

end







function xyw = cub_circsegm(n,omega,r)

%--------------------------------------------------------------------------
% Object:
% The routine computed the nodes and weights of a product gaussian formula
% on a circular segment of a disk centered at the origin
% with angles in [-omega,omega]
%--------------------------------------------------------------------------
% Input:
% n: algebraic degree of exactness
% omega: half-length of the angular interval, 0<omega<=pi
% r: radius of the disk
%--------------------------------------------------------------------------
% Output:
% xyw: (ceil((n+1)/2) x ceil((n+2)/2) x 3 array of (xnodes,ynodes,weights)
%--------------------------------------------------------------------------
% Required routines:
% 1. r_jacobi.m (www.cs.purdue.edu/archives/2002/wxg/codes/OPQ.html)
% 2. gauss.m (www.cs.purdue.edu/archives/2002/wxg/codes/OPQ.html)
% 3. trigauss.m
%--------------------------------------------------------------------------
% Written by Gaspare Da Fies and Marco Vianello, University of Padova
% Date: December 2, 2011.
% Last update: January 4. 2020.
%--------------------------------------------------------------------------

% trigonometric gaussian formula on the arc
tw=trigauss(n+2,-omega,omega);

% algebraic gaussian formula on [-1,1]
ab=r_jacobi(ceil((n+1)/2),0,0);
xw=gauss(ceil((n+1)/2),ab);

% creating the grid
[t,theta]=meshgrid(xw(:,1),tw(1:ceil((n+2)/2),1));
[w1,w2]=meshgrid(xw(:,2),tw(1:ceil((n+2)/2),2));

% nodal cartesian coordinates and weights
s=sin(theta(:));
xyw(:,1)=r*cos(theta(:));
xyw(:,2)=r*t(:).*s;
xyw(:,3)=r^2*s.^2.*w1(:).*w2(:);











function tw=trigauss(n,alpha,beta)

% by Gaspare Da Fies, Alvise Sommariva and Marco Vianello
% University of Padova
% February 28, 2013

% computes the n+1 angles and weights of a trigonometric gaussian
% quadrature formula on [alpha,beta], 0<beta-alpha<=2*pi

% improves a previous version by G. Da Fies and M. Vianello

% input:
% n: trigonometric degree of exactness
% [alpha,beta]: angular interval, 0<beta-alpha<=2*pi

% output:
% tw: (n+1) x 2 array of (angles,weights)

% the formula integrates the canonical trigonometric basis with accuracy
% from about 10^(-15) (small omega) to about 10^(-13) (omega-->pi)
% up to n=300


n=n+1;
omega=(beta-alpha)/2;
ab = r_subchebyshev(n,omega);
xw_symm_eigw = SymmMw(n,ab);
tw=trigauss_conversion(xw_symm_eigw,omega);
tw(:,1)=tw(:,1)+(beta+alpha)/2;




function ab=r_subchebyshev(n,omega)
%% Authors:
%% Gerard Meurant and Alvise Sommariva
%%
%% Date: June 3, 2012
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
% INPUT:
% n    : NUMBER OF POINTS.
% omega: ARC ANGLE.
%
% OUTPUT:
% ab   : COEFFS of THREE TERM RECURSION.
%
%--------------------------------------------------------------------------


N = n;
n = n - 1;


% modified Chebyshev moments by recurrence

if rem(N,2) == 1
    NN=N+1; nn=n+1;
else
    NN=N; nn=n;
end

mom=fast_moments_computation(omega,2*nn+1);

% recurrence coeffs of the monic Chebyshev polynomials
abm(:,1)=zeros(2*nn+1,1);
abm(:,2)=0.25*ones(2*nn+1,1); abm(1,2)=pi; abm(2,2)=0.5;

% recurrence coeffs for the monic OPS w.r.t. the weight function
% w(x)=2*sin(omega/2)/sqrt(1-sin^2(omega/2)*x^2)
% by the modified Chebyshev algorithm

% ab = chebyshev(NN+1,mom,abm);
ab = fast_chebyshev(NN,mom,abm);




function x = tridisolve(a,b,c,d)
%   TRIDISOLVE  Solve tridiagonal system of equations.
% From Cleve Moler's Matlab suite
% http://www.mathworks.it/moler/ncmfilelist.html

%     x = TRIDISOLVE(a,b,c,d) solves the system of linear equations
%     b(1)*x(1) + c(1)*x(2) = d(1),
%     a(j-1)*x(j-1) + b(j)*x(j) + c(j)*x(j+1) = d(j), j = 2:n-1,
%     a(n-1)*x(n-1) + b(n)*x(n) = d(n).
%
%   The algorithm does not use pivoting, so the results might
%   be inaccurate if abs(b) is much smaller than abs(a)+abs(c).
%   More robust, but slower, alternatives with pivoting are:
%     x = T\d where T = diag(a,-1) + diag(b,0) + diag(c,1)
%     x = S\d where S = spdiags([[a; 0] b [0; c]],[-1 0 1],n,n)

% optimized version
x = d;
n = length(x);
bi = zeros(n,1);

for j = 1:n-1
    bi(j) = 1 / b(j);
    mu = a(j) * bi(j);
    b(j+1) = b(j+1) - mu * c(j);
    x(j+1) = x(j+1) - mu * x(j);
end

x(n) = x(n) / b(n);
for j = n-1:-1:1
    x(j) = (x(j) - c(j) * x(j+1)) * bi(j);
end




function ab=fast_chebyshev(N,mom,abm)
%SUBP_MOD_CHEBYSHEV Modified Chebyshev algorithm
% this works only for the subperiodic weight function
%
% From Gautschi's code (simplified)
% Mar 2012
%

ab = zeros(N,2);
sig = zeros(N+1,2*N);

ab(1,2) = mom(1);

sig(1,1:2*N) = 0;
sig(2,:) = mom(1:2*N);

for n = 3:N+1
    for m = n-1:2*N-n+2
        sig(n,m) = sig(n-1,m+1) + abm(m,2) * sig(n-1,m-1) - ab(n-2,2) * sig(n-2,m);
    end

    ab(n-1,2) = sig(n,n-1) / sig(n-1,n-2);
end




function mom=fast_moments_computation(omega,n)

mom=zeros(1,n+1);
mom(1)=2*omega; % FIRST MOMENT.

if(n>=2)

    if(omega<=1/4*pi)
        l=10;
    elseif(omega<=1/2*pi)
        l=20;
    elseif(omega<=3/4*pi)
        l=40;
    else
        if omega == pi
            l=2*ceil(10*pi);
        else
            l=2*ceil(10*pi/(pi-omega));
        end
    end


    temp=(2:2:n+2*l-2); % AUXILIAR VECTORS.
    temp2=temp.^2-1;

    dl=1/4 -1./(4*(temp-1)); % DIAGONALS.
    dc=1/2 -1/sin(omega/2)^2 -1./(2*temp2);
    du=1/4 +1./(4*(temp+1));

    d=4*cos(omega/2)/sin(omega/2)./temp2'; % COMPUTING KNOWN TERM.
    d(end)=d(end);                         % PUT LAST MOMENT NULL.

    z=tridisolve(dl(2:end),dc,du(1:end-1),d); % SOLVE SYSTEM.
    mom(3:2:n+1)=z(1:floor(n/2)); % SET ODD MOMENTS.

end

mom=mom';

normalized = 0;

if normalized == 0
    M=length(mom);
    kk=2.^(-((1:2:M)-2))'; kk(1)=1;
    v=ones(M,1);
    v(1:2:M)=kk;
    mom=v.*mom;
end




function xw=SymmMw(N,ab)
%SymmMw computation of the nodes and weights for a symmetric weight
%function
% this version uses the reduced matrix and eig and
% computation of weights with the 3-term recurrence

%
% see: Fast variants of the Golub and Welsch algorithm for symmetric
% weight functions by G. Meurant and A. Sommariva (2012)

% Input
% N : cardinality of the rule
% ab: 3-term recurrence for the orthogonal polynomials
% same as in OPQ
% ab(1,2) is the 0th moment

% Output
% xw : xw(:,1) nodes, xw(:,2) weights of the quadrature rule
% nloop: number of iterations in QR

%
% Authors G. Meurant and A. Sommariva
% June 2012
%

N0 = size(ab,1);
if N0 < N
    error('SymmMw: input array ab is too short')
end

na = norm(ab(:,1));
if na > 0
    error('SymmMw: the weight function must be symmetric')
end

% computation of the reduced matrix in vectors (a,b)

if mod(N,2) == 0
    even = 1;
    Nc = N / 2;
else
    even = 0;
    Nc = fix(N / 2) +1;
end


absd = ab(:,2);
absq = sqrt(absd);

a = zeros(1,Nc);
b = a;

switch even
    case 1
        % N even
        a(1) = absd(2);
        b(1) = absq(2) * absq(3);

        k = (2:Nc-1);
        a(k) = absd(2*k-1) + absd(2*k);
        b(k) = absq(2*k) .* absq(2*k+1);
        a(Nc) = absd(N) + absd(N-1);
        start = 1;

        J = diag(a) + diag(b(1:Nc-1),1) + diag(b(1:Nc-1),-1);
        t = sort(eig(J));
        w = weights_3t(t',a,b);
        % w are the squares of the first components
        w = w' / 2;
    case 0
        % N odd
        a(1) = absd(2);
        b(1) = absq(2) * absq(3);

        k = (2:Nc-1);
        a(k) = absd(2*k-1) + absd(2*k);
        b(k) = absq(2*k) .* absq(2*k+1);
        a(Nc) = absd(N);
        start = 2;

        % the first node must be zero
        J = diag(a) + diag(b(1:Nc-1),1) + diag(b(1:Nc-1),-1);
        t = sort(eig(J));
        t(1) = 0;
        w = weights_3t(t',a,b);
        w = [w(1); w(2:end)' / 2];
    otherwise
        error('this is not possible')
end

xwp = sqrt(t);

xw(:,1) = [-xwp(end:-1:start,1); xwp];
xw(:,2) = ab(1,2) * ([w(end:-1:start); w]);




function tw=trigauss_conversion(xw,omega)

tw(:,1)=2*asin(sin(omega/2)*xw(:,1));
tw(:,2)=xw(:,2);




function w=weights_3t(t,a,b)
%WEIGHTS_3T squares of the 1st components of eigenvectors from the 3-term
% recurrence relation of the orthogonal polynomials
%

% Input
% t: nodes
% a,b coefficients of the 3-term recurrence
%
% Ouput
% w: squares of the first components of the eigenvectors
%

%
% Authors G. Meurant and A. Sommariva
% June 2012
%

N = length(t);

P = zeros(N,N);
P(1,:) = ones(1,N);
P(2,:) = (t - a(1)) / b(1);

for k = 3:N
    k1 = k - 1;
    k2 = k - 2;
    P(k,:) = ((t - a(k1)) .* P(k1,:) - b(k2) * P(k2,:)) / b(k1);
end

P2 = P .* P;

w = 1 ./ sum(P2);






function ab=r_jacobi(N,a,b)

nu=(b-a)/(a+b+2);
mu=2^(a+b+1)*gamma(a+1)*gamma(b+1)/gamma(a+b+2);
if N==1
    ab=[nu mu]; return
end

N=N-1;
n=1:N;
nab=2*n+a+b;
nuadd=(b^2-a^2)*ones(1,N)./(nab.*(nab+2));
A=[nu nuadd];
n=2:N;
nab=nab(n);
B1=4*(a+1)*(b+1)/((a+b+2)^2*(a+b+3));
B=4*(n+a).*(n+b).*n.*(n+a+b)./((nab.^2).*(nab+1).*(nab-1));
abadd=[mu; B1; B'];
ab=[A' abadd];










