
function xyzw=cub_sphere_low_symm(deg)

%--------------------------------------------------------------------------
% Object:
%--------------------------------------------------------------------------
% Cubature over the unit sphere.
% This routine loads spherical designs computed by Rob Womersley, useful
% for cubature on the unit 2-sphere, with algebraic degree of precision at
% most "deg".
% All the degrees are odd (at most of degree 325).
%
% These formulae are referred as:
% "Symmetric Spherical Designs on the sphere S2 with good geometric 
% properties".
%--------------------------------------------------------------------------
% Input:
%--------------------------------------------------------------------------
% deg: algebraic degree of precision of the cubature rule.
%--------------------------------------------------------------------------
% Output:
%--------------------------------------------------------------------------
% xyzw: nodes and weights are stored in a "M x 4" matrix, where "M" is the
%     cardinality of the rule.
%     The k-th node, in cartesian coordinates correspond to "xyzw(k,1:3)",
%     while the k-th weight is "xyzw(k,4)".
%     The weights are choosen so that their sum is "4*pi".
%--------------------------------------------------------------------------
% Reference:
%--------------------------------------------------------------------------
% All these rules are taken from
% ""
% and are studied and stored by Rob Womersley.
%
% They are stored at the homepage:
%      https://web.maths.unsw.edu.au/~rsw/Sphere/EffSphDes/ss.html
% and organized here for practical usage.
%--------------------------------------------------------------------------
% Rules cardinalities:
%--------------------------------------------------------------------------
%--------------------------------------------------------------------------

% Troubleshooting.
if nargin < 1, deg=16; end

if deg > 325
    fprintf(2,'\n \t The ADE is too high. Setting deg=325.');
    deg=325;
end

% The degrees for these rules are odd. Using the closer and larger one.
deg=2*floor(deg/2)+1;

degstr=num2str(deg);
if deg < 10
    degstr=strcat('00',degstr);
else
    if deg < 100,  degstr=strcat('0',degstr); end
end

% Getting file with rule data.
filename=strcat('ss',degstr,'.dat'); cd('RWS_SYMM');
files=dir(filename);
fName=files.name; fFolder=files.folder;
fullname=fullfile(fFolder,fName);

% Loading data and assemblying results.
xyzwVALS=load(fullname);
xyz=xyzwVALS;
w=(4*pi/size(xyz,1))*ones(size(xyz,1),1);
xyzw=[xyz w];

% Back to main folder.
cd ..