
function demo_FTEST_square

%--------------------------------------------------------------------------
% Object
%--------------------------------------------------------------------------
% Testing function reconstruction by hyperinterpolation or alternative
% least-squares methods, on unit-square [-1,1]^2, based on Gram matrix at
% degree "n" via formulas with "ADE=m" (in particular it may be m <= 2n).
%
% The methods used are
% L : hyperinterpolation via formula with "ADE=m".
% G : least-squares via Gram matrix using f.la with "ADE=m".
% H : hyperinterpolationvia formula with "ADE=2*n".
%
%  ftype:
%
% 1. f=@(x,y) exp(-x.^2-y.^2);
% 2. f=@(x,y) (0.5+x+0.1*y).^a;
% 3. f=@(x,y) sqrt( (x-0.5).^2 + (y-0.5).^2 ).^3;
% 4. f=@(x,y) sqrt( (x-0.5).^2 + (y-0.5).^2 ).^7;
% 5. f=@(x,y) sin(pi*x+pi*y);
%--------------------------------------------------------------------------
% Reference paper:
%--------------------------------------------------------------------------
% "On the role of weak Marcinkiewicz-Zygmund constants in polynomial 
% approximation by orthogonal bases"
% C. An, A. Sommariva and M. Vianello
%--------------------------------------------------------------------------
% License:
%--------------------------------------------------------------------------
% Copyright (C) 2025 Alvise Sommariva, Marco Vianello.
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation; either version 3 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%
% Authors:
%
% Alvise Sommariva <alvise@math.unipd.it>
% Marco Vianello <marcov@math.unipd.it>
% Date: December 27, 2025
%--------------------------------------------------------------------------

clf;

m=15;                   % ADE unfettered hyperinterpolation rule
nH=1:m;                 % Hyperinterpolation degree
mCH=2*m;                % ADE classical hyperinterpolation rule
mR=50;                  % degree of reference rule (L2 norms used for errs)

pset_typeL=2;            % 1. Product-rule,      2. Padua points
pset_typeH=1;                        % 3. Low cardinality,   4. MPX
pset_typeR=1;

ftypeV=[1 2 5 3 4];

domain='square';

save_fig=1;



% ......................... main routine below ............................



% Quadrature rule with "ADE=m" (hyperinterpolation up to degree "m").
ruleL=define_rule(pset_typeL,m);
ptsL=ruleL(:,1:end-1); wL=ruleL(:,end);
cardL=length(wL);

% Quadrature rule with "ADE=2*m" (hyperinterpolation up to degree "m").
ruleH=define_rule(pset_typeH,mCH);
ptsH=ruleH(:,1:end-1); wH=ruleH(:,end);
cardH=length(wH);

% Quadrature rule with "ADE=nR" (reference rule).
ruleR=define_rule(pset_typeR,mR);
ptsR=ruleR(:,1:end-1); wR=ruleR(:,end);


% Function evaluations

fLC=zeros(cardL,length(ftypeV));
fHC=zeros(cardH,length(ftypeV));
fRC=zeros(cardR,length(ftypeV));
norm2f=zeros(length(ftypeV),1);

fstr={'','','','',''};

for i=1:length(ftypeV)

    % analyse functions as batch
    ftype=ftypeV(i);
    [f,fstr{i}]=define_function(ftype,m); % m: ade, used to define polyn.
    fLC(:,i)=f(ptsL(:,1),ptsL(:,2));
    fHC(:,i)=f(ptsH(:,1),ptsH(:,2));
    fRC(:,i)=f(ptsR(:,1),ptsR(:,2));
    norm2f(i)=wR'*(fRC(:,i)).^2;

end




% ............................ make tests .................................

err2L=zeros(length(nH),length(ftypeV));
err2H=err2L; err2G=err2L;

for k=1:length(nH)

    % define function
    n=nH(k);

    VL=vandermonde_orthonormal(n,ptsL);
    VH=vandermonde_orthonormal(n,ptsH);
    VR=vandermonde_orthonormal(n,ptsR);

    LL=size(VL,2);
    LH=size(VH,2);

    for i=1:length(ftypeV)

        % Hyper: low degree rule
        fL=fLC(:,i);
        coeffL_lowdim=(fL.*wL)'*VL; coeffL_lowdim=coeffL_lowdim';
        coeffL=zeros(LH,1); coeffL(1:LL,1)=coeffL_lowdim;

        % Hyper: high degree rule
        fH=fHC(:,i);
        coeffH=(fH.*wH)'*VH; coeffH=coeffH';

        % GRAM : low degree rule
        coeffG=hypGRAM(VL',fL,wL);

        % Evaluating errors
        fR=fRC(:,i);

        pL_XR=VR*coeffL;
        pH_XR=VR*coeffH;
        pG_XR=VR*coeffG;

        err2L(i,k)=sqrt(wR'*((fR-pL_XR).^2))/norm2f(i);  % HYP : low ade, low n
        err2H(i,k)=sqrt(wR'*((fR-pH_XR).^2))/norm2f(i);  % HYP : high ade, high n
        err2G(i,k)=sqrt(wR'*((fR-pG_XR).^2))/norm2f(i);  % GRAM: high ade, high n

        fprintf('\n \t n: %-2.0f f: %-2.0f err2L: %1.3e  err2H: %1.3e  err2G: %1.3e ', ...
            n,i,err2L(i,k),err2H(i,k),err2G(i,k));

    end
end


% ............................ make plots .................................

% Hyper.: ade=m

mlim=10^(floor(log10(max([min(min(err2G)) 10^(-20)]))));
Mlim=10^(ceil(log10(max(max(err2G)))));

figure(1)
title_str=['Hyp. with ade ',num2str(m)];
plot_fig(ftypeV,pset_typeL,nH,err2L,mlim,Mlim,m,fstr,domain,'L',...
    save_fig,title_str);

% Hyper.: ade=2*m
figure(2)
title_str=['Hyp. with ade ',num2str(mCH)];
plot_fig(ftypeV,pset_typeH,nH,err2H,mlim,Mlim,m,fstr,domain,'H',...
    save_fig,title_str);

% Gram:  ade=m
figure(3)
title_str=['LS with ade ',num2str(m)];
plot_fig(ftypeV,pset_typeL,nH,err2G,mlim,Mlim,m,fstr,domain,'G',...
    save_fig,title_str);
hold off;










%--------------------------------------------------------------------------
% Subroutines
%--------------------------------------------------------------------------

function [f,str]=define_function(ftype,a)

if nargin < 2, a=1; end

switch ftype
    case 1 % Gaussian: analytic
        str='exp(-x^2-y^2)';
        f=@(x,y) exp(-x.^2-y.^2);

    case 2 % Polynomial
        str=['(0.5+x+0.1*y)^{',num2str(a),'}'];
        f=@(x,y) (0.5+x+0.1*y).^a;

    case 3 % Low regularity at "0.5".
        str='d((x,y),(0.5,0.5))^{3}';
        f=@(x,y) sqrt( (x-0.5).^2 + (y-0.5).^2 ).^3;

    case 4 % Low regularity at "0.5".
        str='d((x,y),(0.5,0.5))^{7}';
        f=@(x,y) sqrt( (x-0.5).^2 + (y-0.5).^2 ).^7;

    case 5 % Regular with oscillations
        str='sin(\pi*x+\pi*y)';
        f=@(x,y) sin(pi*x+pi*y);

end









function save_figure(domain,mode_str,pset_type,m,save_fig)

if save_fig == 1
    filename=['figure_',domain,'_ftest_',mode_str,'_'...
        num2str(pset_type),'_',num2str(max(m))];
    file_fig=[filename '.fig'];
    savefig(file_fig);
    file_eps=[filename '.eps'];
    saveas(gca,file_eps,'epsc');
end









function plot_fig(ftypeV,pset_type,nH,err2L,mlim,Mlim,m,fstr,domain,...
    meth_str,save_fig,title_str)

lw=2;

for i=1:length(ftypeV)
    semilogy(nH,err2L(i,:),'-','LineWidth',lw);
    hold on;
    colororder("gem12");
end

ylim([mlim,Mlim]);

grid on;

xticks(1:nH(end));

title(title_str);
lgd=legend(fstr{1},fstr{2},fstr{3},fstr{4},fstr{5},'Location','southwest');
fontsize(lgd,7,'points')
save_figure(domain,meth_str,pset_type,m,save_fig);
hold off;









%--------------------------------------------------------------------------
% Specialized subroutines
%--------------------------------------------------------------------------

function V=vandermonde_orthonormal(n,pts,pset_type)

if nargin < 3, pset_type=1; end

switch pset_type
    case 4
        V=dCHEBVAND_orthn(n,pts,[-1 -1; 1 1]);
    otherwise
        V=vandermonde_nlegendre_orthn(n,pts,[-1 -1; 1 1]);
end









function rule=define_rule(pset_type,deg)

switch pset_type

    case 1 % Product rule basis
        rule=cub_square_prodrule(deg,[0 0; 0 0]);

    case 2 % Cubature based on Padua Points
        rule = cub_square_padua(deg,[-1 1 -1 1]);

    case 3 % Low cardinality rule
        rule=cub_square_lowcard(deg);

    case 4 % MPX rule
        rule=cub_square_mpx(deg);
        rule(:,3)=pi^2*rule(:,3);
end

