
function demo_MZ_square(mV,nV,pset_typeV,show_stats,save_fig,save_tab)

%--------------------------------------------------------------------------
% Object
%--------------------------------------------------------------------------
% Computation of Marcinkiewicz-Zygmund constants on the unit-square.
% It performs all the tests and provide results.
%--------------------------------------------------------------------------
% Reference paper:
%--------------------------------------------------------------------------
% "On the role of weak Marcinkiewicz-Zygmund constants in polynomial 
% approximation by orthogonal bases"
% C. An, A. Sommariva and M. Vianello
%--------------------------------------------------------------------------
% License:
%--------------------------------------------------------------------------
% Copyright (C) 2025 Alvise Sommariva, Marco Vianello.
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation; either version 3 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%
% Authors:
%
% Alvise Sommariva <alvise@math.unipd.it>
% Marco Vianello <marcov@math.unipd.it>
% Date: December 27, 2025
%--------------------------------------------------------------------------

domain='square';

if nargin < 1, mV=1:20; end
if nargin < 2, nV=0:30; end
if nargin < 3, pset_typeV=1:4; end
if nargin < 4, show_stats=0; end % show temporary statistics
if nargin < 5, save_fig=0; end
if nargin < 6, save_tab=0; end



% -------------------------- Main code below ------------------------------

% Make all make_tests
for k=1:length(pset_typeV)
    pset_type=pset_typeV(k);

    if show_stats
        fprintf('\n \t ................ %2.0f .................. \n \n',...
            pset_type);
    end
    [table_data{k},A_mat{k},B_mat{k},cond_mat{k}]=make_test(mV,nV,...
        pset_type,show_stats,save_fig,save_tab,domain);

end

% Assembly summary table.
if show_stats
    assembly_latex_table(table_data,mV);
end








function [table_data,A_mat,B_mat,cond_mat]=make_test(mV,nV,pset_type,...
    show_stats,save_fig,save_tab,domain)

% ---------------------- Compute MZ constants -----------------------------

eta_mat=zeros(length(mV),length(nV));
A_mat=eta_mat;
B_mat=eta_mat;
cond_mat=eta_mat;
mV_mat=eta_mat;
nV_mat=eta_mat;



for i=1:length(mV)

    m=mV(i);
    rule=define_rule(pset_type,m);
    w=rule(:,end);

    for j=1:length(nV)
        n=nV(j);

        mV_mat(i,j)=m; nV_mat(i,j)=n;

        V=vandermonde_orthonormal(n,rule(:,1:end-1),pset_type);
        [eta_mat(i,j),G,~,A_mat(i,j),B_mat(i,j)]=compute_MZ_constant(V',w);
        cond_mat(i,j)=cond(G);

        if show_stats
            fprintf('\n \t m: %-2.0f n: %-2.0f eta: %1.2e A: %1.2e B: %1.2e cond: %1.2e',...
                m,n,eta_mat(i,j),A_mat(i,j),B_mat(i,j),cond_mat(i,j));
        end

    end

end


if show_stats
    fprintf('\n \n');
end

% ---------------------------- Statistics ---------------------------------

table_data=print_MZ_statistics(mV,nV,eta_mat,show_stats);

% ----------------------------- Anomalies ---------------------------------

if show_stats
    print_MZ_anomalies(mV,nV,table_data,A_mat,B_mat,eta_mat);
    print_cond_anomalies(mV_mat,nV_mat,eta_mat,cond_mat);
end

% ............................ save MZ matrix .............................

if save_tab
    save_data(domain,pset_type,A_mat,B_mat,cond_mat);
end

% -------------------------- Plot results ---------------------------------

plot_MZ_figures(mV,nV,eta_mat,A_mat,B_mat,cond_mat,domain,pset_type,...
    save_fig);

if show_stats
    fprintf('\n \n');
end


















%--------------------------------------------------------------------------
% Subroutines
%--------------------------------------------------------------------------


function V=vandermonde_orthonormal(n,pts,pset_type)

if nargin < 3, pset_type=1; end

switch pset_type
    case 4
        V=dCHEBVAND_orthn(n,pts,[-1 -1; 1 1]);
    otherwise
        V=vandermonde_nlegendre_orthn(n,pts,[-1 -1; 1 1]);
end




function rule=define_rule(pset_type,deg)

switch pset_type

    case 1 % Product rule basis
        rule=cub_square_prodrule(deg,[0 0; 0 0]);

    case 2 % Cubature based on Padua Points
        rule = cub_square_padua(deg,[-1 1 -1 1]);

    case 3 % Low cardinality rule
        rule=cub_square_lowcard(deg);

    case 4 % MPX rule
        rule=cub_square_mpx(deg);
        rule(:,3)=pi^2*rule(:,3);

    case 5 % QMC rule
        card=2^deg;
        dbox=[-1 -1; 1 1];
        dim=2;
        rule=cub_square_QMC(card,dbox,dim);
end

