
function V=vandermonde_dubiner_legendre(n,pts)

%--------------------------------------------------------------------------
% OBJECT.
%--------------------------------------------------------------------------
% Vandermonde matrix w.r.t. Legendre measure on the triangle [0 0;1 0;0 1]
% * of degree "n"
% * evaluated at the pointset "pts" (N x 2 matrix of cartesian
%   coordinates).
% 
% Note: the orthonormality of the basis has been checked numerically.
%--------------------------------------------------------------------------
% FUNCTIONS USED.
%--------------------------------------------------------------------------
% 1. r_jacobi
% 2. eval_mod_op
% 3. eval_op
%--------------------------------------------------------------------------
%% Copyright (C) 2010. Alvise Sommariva, Marco Vianello.
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Author:  Alvise Sommariva <alvise@euler.math.unipd.it>
%%          Marco Vianello   <marcov@euler.math.unipd.it>
%%
%% Date: October 18, 2010
%% Modified: December 5, 2025.
%--------------------------------------------------------------------------

x=pts(:,1); y=pts(:,2);

c=1-y;
t=2*x-c; 

a1=0; b1=0;

ab=r_jacobi(n,a1,b1);
V1=eval_mod_op(n,ab,a1,b1,t,c,4);

t=2*y-1; V=[];

for h=0:n
    
    a2=2*h+1; b2=0;
    
    if n-h > 0
        ab2=r_jacobi(n-h,a2,b2);    
    else
        ab2=[];
    end
    
    V2loc=eval_op(n-h,ab2,t,a2,b2,3);
    V1loc=repmat(V1(:,h+1),1,size(V2loc,2));
    V=[V V1loc.*V2loc];
    
end 



function V=eval_op(deg,ab,x,a,b,type)

%%
%% Copyright (C) 2010. Alvise Sommariva, Marco Vianello.
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Author:  Alvise Sommariva <alvise@euler.math.unipd.it>
%%          Marco Vianello   <marcov@euler.math.unipd.it>
%%
%% Date: October 18, 2010
%%

if nargin < 3
    a=0; b=0;
end

L=size(x,1);

if deg > 0
  aa=ab(:,1);
  bb=ab(:,2);
else
    ab=r_jacobi(1,a,b);
    aa=[];
    bb=[];
end

switch type
case 1 % MONIC JACOBI POLYNOMIALS.
    V=[zeros(L,1) ones(L,1)];
    cn=ones(size(ab,1),1);
    co=ones(size(ab,1),1);
    
case 2 % CLASSICAL JACOBI (SEE WOLFRAM HOMESITE)
    V=[zeros(L,1) ones(L,1)];
    [cn,co]=c_standard(deg,a,b);
    
case 3 % ORTHONORMAL JACOBI POLYNOMIALS.
    V=[zeros(L,1) (1/sqrt(ab(1,2)))*ones(L,1)]; % "ab(1,2)" IS "int_-1^1 w(x) dx".
    [cn,co]=c_orthnrm(deg,a,b);
end

for ii=1:deg
    Vloc=cn(ii)*(x-aa(ii)).*V(:,ii+1)-co(ii)*bb(ii)*V(:,ii);
    V=[V Vloc];
end

V=V(:,2:end);



function V=eval_mod_op(deg,ab,a,b,t,c,type)

%--------------------------------------------------------------------------
%%
%% Copyright (C) 2010. Alvise Sommariva, Marco Vianello.
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Author:  Alvise Sommariva <alvise@euler.math.unipd.it>
%%          Marco Vianello   <marcov@euler.math.unipd.it>
%%
%% Date: October 18, 2010
%%
%--------------------------------------------------------------------------

if nargin < 3
    a=0; b=0;
end

L=size(t,1);
aa=ab(:,1);
bb=ab(:,2);

switch type
case 1 % MONIC JACOBI POLYNOMIALS.
    V=[zeros(L,1) ones(L,1)];
    cn=ones(size(ab,1),1);
    co=ones(size(ab,1),1);
case 2 % CLASSICAL JACOBI (SEE WOLFRAM HOMESITE)
    V=[zeros(L,1) ones(L,1)];
    [cn,co]=c_standard(deg,a,b);
    
case 3 % ORTHONORMAL JACOBI POLYNOMIALS.
    V=[zeros(L,1) (1/sqrt(ab(1,2)))*ones(L,1)]; % "ab(1,2)" IS "int_-1^1 w(x) dx".
    [cn,co]=c_orthnrm(deg,a,b);
    
case 4 % ORTHONORMAL JACOBI POLYNOMIALS USED BY DUBINER ORTHONORMAL BASIS OVER THE SYMPLEX.
    V=[zeros(L,1) (sqrt(8)/sqrt(ab(1,2)))*ones(L,1)]; % "ab(1,2)" IS "int_-1^1 w(x) dx".
    [cn,co]=c_dub(deg,a,b);
    
end

for ii=1:deg
    Vloc=cn(ii)*(t-c.*aa(ii)).*V(:,ii+1)-(c.^2)*co(ii)*bb(ii).*V(:,ii);
    V=[V Vloc];
end

V=V(:,2:end);




function [cn,co]=c_dub(ade,a,b)

[cn_orth,co_orth]=c_orthnrm(ade,a,b);
cn=2*cn_orth;
co=[1; cn(2:end,:).*cn(1:end-1,:)];




function [cn,co]=c_orthnrm(ade,a,b)

[cn_st,co_st]=c_standard(ade,a,b);

ab=r_jacobi(1,a,b);
an0=ab(1,2);

n=(1:ade)';

if length(n) > 0
    if abs(b) > 0 
        anloc=(2^(a+b+1)).*gamma(n+a+1).*...
            gamma(n+b+1)./((2*n+a+b+1).*gamma(n+a+b+1).*gamma(n+1));
    else
        anloc=2^(a+b+1)./(2*n+a+b+1);
    end
else
    anloc=[];
end

an=[an0; anloc];

an=an.^(-1/2);
an_rat=an(2:end,:)./an(1:end-1,:);
cn=cn_st.*an_rat;
co=[1; cn(2:end,:).*cn(1:end-1,:)];




function [cn,co]=c_standard(deg,a,b)

s=a+b;

% INITIALIZATION TO AVOID SOME POSSIBLE OVERFLOW PROBLEMS FOR NEGATIVE s.
if (deg > 0)
    cn1=0.5*gamma(2+s+1)/gamma(s+2);
else
    cn1=[];
end

n=2:deg; n=n';

if (abs(s-floor(s)) > 0) 
    cnloc=(1./(2*n)).*(gamma(2*n+s+1)./gamma(2*n+s-1)).*(gamma(n+s)./gamma(n+s+1));
else
    cnloc=(1./(2*n)).*(2*n+s).*(2*n+s-1)./(n+s);
end
 
cn=[cn1; cnloc];

co=[1; cn(2:end,:).*cn(1:end-1,:)];


%--------------------------------------------------------------------------
% ROUTINE BY W. GAUTSCHI AND D. LAURIE.
%--------------------------------------------------------------------------

function ab=r_jacobi(N,a,b)

if N == 0, N=1; end

nu=(b-a)/(a+b+2);
mu=2^(a+b+1)*gamma(a+1)*gamma(b+1)/gamma(a+b+2);
if N==1
 ab=[nu mu]; return
end

N=N-1;
n=1:N;
nab=2*n+a+b;
nuadd=(b^2-a^2)*ones(1,N)./(nab.*(nab+2));
A=[nu nuadd];
n=2:N;
nab=nab(n);
B1=4*(a+1)*(b+1)/((a+b+2)^2*(a+b+3));
B=4*(n+a).*(n+b).*n.*(n+a+b)./((nab.^2).*(nab+1).*(nab-1));
abadd=[mu; B1; B'];
ab=[A' abadd];

