
function demo_cub_ellblend(domain_example)

%--------------------------------------------------------------------------
% Object:
%--------------------------------------------------------------------------
% Demo on algebraic polynomial cubature on linear blending of elliptical
% arcs.
%
% This routine tests the integration of some test functions on specific
% domains.
%--------------------------------------------------------------------------
% Reference paper
%--------------------------------------------------------------------------
% Algebraic cubature by linear blending of elliptical arcs,
% Applied Numerical Mathematics, 74 (2013), 49-61. (Open Access)
% G. Da Fies, A. Sommariva and M. Vianello.
%--------------------------------------------------------------------------
% Dates:
%--------------------------------------------------------------------------
% Written on January 4, 2026 (by A. Sommariva and M. Vianello).
%--------------------------------------------------------------------------
%% Copyright (C) 2026-
%% Alvise Sommariva, Marco Vianello.
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Authors:
%% Alvise Sommariva, Marco Vianello.
%%
%% Date       : JANUARY 4, 2026
%% Modified on: JANUARY 5, 2026
%--------------------------------------------------------------------------

if nargin < 1, domain_example=1; end
function_example=1;

ade=0;
adeR=20;




%-------------------------- main code below -------------------------------

switch function_example
    case 1
        f=@(x,y) 1./(1+25*(x+2*y).^2);
    case 2
        f=@(x,y) sin(x+2*y);
    case 3
        f=@(x,y) (0.1+0.5*x+0.2*y).^deg;
end

switch domain_example

    case 1 % circular segment

        R=1;
        alpha=0; beta=pi/3;
        A(1,:)=[R 0]; B(1,:)=[0 R]; C(1,:)=[0 0];
        A(2,:)=[R 0]; B(2,:)=[0 -R]; C(2,:)=[0 0];

    case 2 % circular segment

        R=1;
        alpha=-pi/3; beta=pi/3;
        A(1,:)=[0 0]; B(1,:)=[0 R]; C(1,:)=[R*cos(beta) 0];
        A(2,:)=[R 0]; B(2,:)=[0 R]; C(2,:)=[0 0];

    case 3 % circular segment

        R=1;
        alpha=-pi/3; beta=pi/3;
        A(1,:)=[0 0]; B(1,:)=[0 0]; C(1,:)=[R*cos(beta) 0];
        A(2,:)=[R 0]; B(2,:)=[0 R]; C(2,:)=[0 0];

    case 4 % circular-annular-sector

        % Circular annular sector with center (a,b) and radii r1<r2
        % entire DISK: r1=0, alpha=-pi, beta=pi
        % A=[r2 0;r1 0]; B=[0 r2;0 r1]; C=[a b;a b];

        r1=1; r2=2; % r1 < r2
        a=0; b=1;
        alpha=-pi; beta=pi;

        A=[r2 0;r1 0]; 
        B=[0 r2;0 r1]; 
        C=[a b;a b];

    case 5 % disk as circular annular sector

        r1=0; r2=1;
        a=0; b=1;
        alpha=-pi; beta=pi;

        A=[r2 0;r1 0]; 
        B=[0 r2;0 r1]; 
        C=[a b;a b];

    case 6 % sector

        % sector: r1=0, r2=r; alpha>=-pi, beta<=pi
        % A=[r2 0;r1 0]; B=[0 r2;0 r1]; C=[a b;a b];

        r1=0; r2=1; 
        a=0; b=1;
        alpha=-pi/4; beta=pi/6;

        A=[r2 0; r1 0]; 
        B=[0 r2; 0 r1]; 
        C=[a b; a b];

    case 7 % asymmetric-circular-sector

        % Asymmetric-circular-sector with vertex in (c,d), of a disk with
        % center in (a,b) and radius r.
        % A=[r 0;0 0]; B=[0 r;0 0]; C=[a b;c d];

        r=3;
        a=1; b=1; c=2; d=2; 
        alpha=-pi/4; beta=pi/4;

        A=[r 0; 0 0]; 
        B=[0 r; 0 0]; 
        C=[a b; c d];

    case 8 % asymmetric-annulus

        % Asymmetric-annulus (difference of a disk with center in (a,b) and
        % radius r1 with an inner disk with center in (c,d) and radius r2)
        % alpha=-pi, beta=pi;
        % A=[r1 0;r2 0]; B=[0 r1;0 r2]; C=[a b;c d];

        r1=3; r2=4;
        a=1; b=1; c=2; d=2; 
        alpha=-pi; beta=pi;

        A=[r1 0; r2 0]; 
        B=[0 r1;0 r2]; 
        C=[a b;c d];
        

    case 9 % vertical-circular-zone

        % Vertical-circular-zone of a disk with center in (a,b) and
        % radius r.
        %
        % entire DISK: alpha=-pi, beta=pi
        % A=[r 0; r 0]; B=[0 r;0 -r]; C=[a b;a b]; alpha=-beta

        r=1; a=0; b=1;
        alpha=-pi/4; beta=-alpha;

        A=[r 0;r 0]; B=[0 r;0 -r]; C=[a b;a b];

    case 10 % horizontal-circular-zone

        % Horizontal-circular-zone  of a disk with center in (a,b) and
        % radius r.
        %
        % entire DISK: alpha=-pi, beta=pi
        % A=[r 0;-r 0]; B=[0 r;0 r]; C=[a b;a b]; alpha=-beta

        r=1;
        a=0; b=1;
        alpha=-pi/4;
        beta=-alpha;

        A=[r 0;-r 0]; B=[0 r;0 r]; C=[a b;a b];


    case 11 % circular-segment

        % Circular-segment of a disk with center in (a,b) and radius r
        % A=[r 0;r*cos(alpha+beta) r*sin(alpha+beta)];
        % B=[0 r;r*sin(alpha+beta) -r*cos(alpha+beta)]; C=[a b;a b];
      
        r=1;
        a=0; b=1;
        alpha=pi/3; beta=pi/2;

        A=[r 0; r*cos(alpha+beta) r*sin(alpha+beta)];
        B=[0 r; r*sin(alpha+beta) -r*cos(alpha+beta)]; 
        C=[a b;a b];


    case 12 % symmetric-lens

        % Symmetric-lens (intersection of two disks with radius r and 
        % centers in (a,0) and (-a,0), a<r)
        % beta=acos(a/r);alpha=-beta;
        % A=[r 0;-r 0]; B=[0 r;0 r]; C=[-a 0;a 0];

        r=1;
        a=0.5; % a<r
        beta=acos(a/r); alpha=-beta;

        A=[r 0;-r 0]; 
        B=[0 r;0 r]; 
        C=[-a 0;a 0];


    case 13 % butterfly

        % Butterfly domain: linear blending of two opposite circular arcs
        % of a disk with center in (a,b) and radius r
        % A=[r 0;-r 0]; B=[0 r;0 -r]; C=[a b;a b];

        a=0; b=1;
        r=1;
        alpha=-pi; beta=pi;

        A=[r 0;-r 0]; B=[0 r;0 -r]; C=[a b;a b];


    case 14 % candy

        % Candy domain: linear blending of two overlapping circular arcs
        % with opposite concavity, beta>acos(a/r), a<r, alpha=-beta
        % A=[r 0;-r 0]; B=[0 r;0 r]; C=[-a 0;a 0];
        %
        % In this case "domain_parms" is a vector [a r alpha]
        % Example:
        %       a=0.5; r=1; alpha=-1.5;
        %       domain_struct.parms=[a r alpha];

        a=0.5; 
        r=1; 
        alpha=-1.5; beta=-alpha;

        acos_a_r=acos(a/r);
        if beta <= acos_a_r
            error('it must be -alpha > acos(a/r)');
        end
        A=[r 0;-r 0]; B=[0 r;0 r]; C=[-a 0;a 0];


end


%-------------------------- cubature rule ---------------------------------

xyw = cub_ellblend(ade,A,B,C,alpha,beta);
x=xyw(:,1); y=xyw(:,2); w=xyw(:,3);
I=w'*f(x,y);

%-------------------------- ref cubature rule -----------------------------

xywR = cub_ellblend(adeR,A,B,C,alpha,beta);
xR=xywR(:,1); yR=xywR(:,2); wR=xywR(:,3);
IR=wR'*f(xR,yR);

%---------------------------- statistics ----------------------------------

fprintf('\n \t ade  : %-2.0f',ade);
fprintf('\n \t card : %-2.0f',length(w));

fprintf('\n \n \t intg     : %-1.15e',I);
fprintf('\n \t intg ref : %-1.15e',IR);

fprintf('\n \n \t abs.err. : %-1.15e',abs(I-IR));
fprintf('\n \t rel.err. : %-1.15e',abs(I-IR)/abs(IR));

fprintf('\n \n');

%------------------------------ plot --------------------------------------

fig1=figure(1);
clf(fig1,'reset');

% ........................... plot domain .................................

theta=linspace(alpha,beta,1000); theta=theta';

% arc P
A1=A(1,:); B1=B(1,:); C1=C(1,:);

xx1=A1(1)*cos(theta)+B1(1)*sin(theta)+C1(1);
yy1=A1(2)*cos(theta)+B1(2)*sin(theta)+C1(2);

xx1A=xx1(1); xx1B=xx1(end); yy1A=yy1(1); yy1B=yy1(end); 

% arc Q
A2=A(2,:); B2=B(2,:); C2=C(2,:);

xx2=A2(1)*cos(theta)+B2(1)*sin(theta)+C2(1);
yy2=A2(2)*cos(theta)+B2(2)*sin(theta)+C2(2);

xx2A=xx2(1); xx2B=xx2(end); yy2A=yy2(1); yy2B=yy2(end); 

plot(xx1,yy1,'k-','LineWidth',2); % arc P
hold on;
plot(xx2,yy2,'k-','LineWidth',2); % arc Q

plot([xx1A; xx2A],[yy1A,yy2A],'k-','LineWidth',2); % segment 1

plot([xx1B; xx2B],[yy1B,yy2B],'k-','LineWidth',2); % segment 2


% ........................... plot pointset ...............................

plot(x,y,'o','MarkerFaceColor','red','MarkerSize',3);

axis equal;
title('Linear blending of elliptical arcs');

hold off;









%--------------------------------------------------------------------------
% Examples
%--------------------------------------------------------------------------
% >> % demo_cub_ellblend(1)
% 
%  	 ade  : 10
%  	 card : 78
%  	 intg     : 1.061710284053736e-01
%  	 intg ref : 1.102687146409197e-01
%  	 abs.err. : 4.097686235546169e-03
%  	 rel.err. : 3.716091412591432e-02
% 
% >> % demo_cub_ellblend(2)
% 
%  	 ade  : 10
%  	 card : 78
%  	 intg     : 1.097679707068018e-01
%  	 intg ref : 1.106977081612529e-01
%  	 abs.err. : 9.297374544510045e-04
%  	 rel.err. : 8.398886209068215e-03
% 
% >> % demo_cub_ellblend(3)
% 
%  	 ade  : 10
%  	 card : 72
%  	 intg     : 1.080730273137796e-01
%  	 intg ref : 1.108870338047026e-01
%  	 abs.err. : 2.814006490923004e-03
%  	 rel.err. : 2.537723658366688e-02
% 
% >> % demo_cub_ellblend(4)
% 
%  	 ade  : 10
%  	 card : 72
%  	 intg     : 6.263261427547155e-01
%  	 intg ref : 7.565682858342233e-01
%  	 abs.err. : 1.302421430795078e-01
%  	 rel.err. : 1.721485628172973e-01
% 
% >> % demo_cub_ellblend(5)
% 
%  	 ade  : 10
%  	 card : 72
%  	 intg     : 2.153632238807523e-01
%  	 intg ref : 2.229030592917757e-01
%  	 abs.err. : 7.539835411023432e-03
%  	 rel.err. : 3.382562552070645e-02
%
% >> % demo_cub_ellblend(6)
% 
%  	 ade  : 10
%  	 card : 66
%  	 intg     : 5.077140336419142e-03
%  	 intg ref : 5.077140331970790e-03
%  	 abs.err. : 4.448351409447326e-12
%  	 rel.err. : 8.761529362180566e-10
% 
% >> % demo_cub_ellblend(7)
% 
%  	 ade  : 10
%  	 card : 72
%  	 intg     : 1.172332928475565e-02
%  	 intg ref : 1.172476092804222e-02
%  	 abs.err. : 1.431643286578174e-06
%  	 rel.err. : 1.221042625401512e-04
% 
% >> % demo_cub_ellblend(8)
% 
%  	 ade  : 10
%  	 card : 78
%  	 intg     : 4.209715077460712e-01
%  	 intg ref : 2.604695888928698e-01
%  	 abs.err. : 1.605019188532014e-01
%  	 rel.err. : 6.162021429657775e-01
% 
% >> % demo_cub_ellblend(9)
% 
%  	 ade  : 10
%  	 card : 78
%  	 intg     : 3.424767236877739e-03
%  	 intg ref : 3.424769636057721e-03
%  	 abs.err. : 2.399179981418809e-09
%  	 rel.err. : 7.005376233656765e-07
% 
% >> % demo_cub_ellblend(10)
% 
%  	 ade  : 10
%  	 card : 78
%  	 intg     : 1.034620252822807e-01
%  	 intg ref : 1.036589394072955e-01
%  	 abs.err. : 1.969141250148659e-04
%  	 rel.err. : 1.899634765132539e-03
% 
% >> % demo_cub_ellblend(11)
% 
%  	 ade  : 10
%  	 card : 78
%  	 intg     : 5.481896962458495e-05
%  	 intg ref : 5.481896962458486e-05
%  	 abs.err. : 9.486769009248164e-20
%  	 rel.err. : 1.730563174429605e-15
% 
% >> % demo_cub_ellblend(12)
% 
%  	 ade  : 10
%  	 card : 78
%  	 intg     : 2.612405596723625e-01
%  	 intg ref : 2.616149319010411e-01
%  	 abs.err. : 3.743722286786433e-04
%  	 rel.err. : 1.431004820551504e-03
% 
% >> % demo_cub_ellblend(13)
% 
%  	 ade  : 10
%  	 card : 72
%  	 intg     : 4.859663726860152e-01
%  	 intg ref : 4.508206811937984e-01
%  	 abs.err. : 3.514569149221680e-02
%  	 rel.err. : 7.795935936024286e-02
% 
% >> % demo_cub_ellblend(14)
% 
%  	 ade  : 10
%  	 card : 78
%  	 intg     : 2.406271116115233e-01
%  	 intg ref : 2.659326271686545e-01
%  	 abs.err. : 2.530551555713118e-02
%  	 rel.err. : 9.515761877944527e-02
% 
