
function demo_cub_lens

%--------------------------------------------------------------------------
% Object:
%--------------------------------------------------------------------------
% Demo on algebraic polynomial cubature on circular lens.
%
% This routine tests the integration of some test functions on specific 
% domains.
%--------------------------------------------------------------------------
% Reference paper
%--------------------------------------------------------------------------
% Algebraic cubature by linear blending of elliptical arcs,
% Applied Numerical Mathematics, 74 (2013), 49-61. (Open Access)
% G. Da Fies, A. Sommariva and M. Vianello.
%--------------------------------------------------------------------------
% Dates:
%--------------------------------------------------------------------------
% Written on January 4, 2026 (by A. Sommariva and M. Vianello).
%--------------------------------------------------------------------------
%% Copyright (C) 2026-
%% Alvise Sommariva, Marco Vianello.
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Authors:
%% Alvise Sommariva, Marco Vianello.
%%
%% Date       : JANUARY 4, 2026
%% Modified on: JANUARY 4, 2026
%--------------------------------------------------------------------------

ade=10;
adeR=20;
function_example=1;

x1=0; y1=0; r1=1; 
x2=0.5; y2=0; r2=1; 


%-------------------------- main code below -------------------------------

switch function_example
    case 1
        f=@(x,y) 1./(1+25*(x+2*y).^2);
    case 2
        f=@(x,y) sin(x+2*y);
    case 3
        f=@(x,y) (0.1+0.5*x+0.2*y).^deg;
end

%-------------------------- cubature rule ---------------------------------

xyw = cub_lens(ade,x1,y1,r1,x2,y2,r2);
x=xyw(:,1); y=xyw(:,2); w=xyw(:,3);

I=w'*f(x,y);

%-------------------------- ref cubature rule -----------------------------

xywR = cub_lens(adeR,x1,y1,r1,x2,y2,r2);
xR=xywR(:,1); yR=xywR(:,2); wR=xywR(:,3);
IR=wR'*f(xR,yR);

%---------------------------- statistics ----------------------------------

fprintf('\n \t ade  : %-2.0f',ade);
fprintf('\n \t card : %-2.0f',length(w));

fprintf('\n \n \t intg     : %-1.15e',I);
fprintf('\n \t intg ref : %-1.15e',IR);

fprintf('\n \n \t abs.err. : %-1.3e',abs(I-IR));
fprintf('\n \t rel.err. : %-1.3e',abs(I-IR)/abs(IR));

fprintf('\n \n');

fprintf('\n \n');

%------------------------------ plot --------------------------------------

figure(1)

% circle r1
thetaC=linspace(0,2*pi,1000);
xx=x1+r1*cos(thetaC); yy=y1+r1*sin(thetaC);
plot(xx,yy,'k-','LineWidth',1);

hold on;

% circle r2
xx=x2+r2*cos(thetaC); yy=y2+r2*sin(thetaC);
plot(xx,yy,'k-','LineWidth',1);

% plot pointset
plot(x,y,'o','MarkerFaceColor','red','MarkerSize',3);

axis equal;

title('Circular lens');

hold off;






