
function demo_cub_multibubble

%--------------------------------------------------------------------------
% Object:
%--------------------------------------------------------------------------
% Demo on algebraic polynomial cubature on union of disks.
%
% This routine tests the integration of some test functions on specific
% domains.
%--------------------------------------------------------------------------
% Reference papers
%--------------------------------------------------------------------------
% Algebraic cubature by linear blending of elliptical arcs,
% Applied Numerical Mathematics, 74 (2013), 49-61. (Open Access)
% G. Da Fies, A. Sommariva and M. Vianello.
%
% Numerical quadrature on the intersection of planar disks,
% FILOMAT 31:13 (2017), 4105--4115 (open access)
% A. Sommariva and M. Vianello.
%--------------------------------------------------------------------------
% Dates:
%--------------------------------------------------------------------------
% Written on January 4, 2026 (by A. Sommariva and M. Vianello).
%--------------------------------------------------------------------------
%% Copyright (C) 2026-
%% Alvise Sommariva, Marco Vianello.
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Authors:
%% Alvise Sommariva, Marco Vianello.
%%
%% Date       : JANUARY 4, 2026
%% Modified on: JANUARY 4, 2026
%--------------------------------------------------------------------------

ade=10;
adeR=20;
function_example=1;


%-------------------------- main code below -------------------------------

% Domain
numdisks=3;

cent=rand(numdisks,2);
r=rand(numdisks,1);

% Integrands

switch function_example
    case 1
        f=@(x,y) 1./(1+25*(x+2*y).^2);
    case 2
        f=@(x,y) sin(x+2*y);
    case 3
        f=@(x,y) (0.1+0.5*x+0.2*y).^deg;
end

%-------------------------- cubature rule ---------------------------------


xyw = cub_multibubble(ade,cent,r);

if isempty(xyw)
    fprintf('\n \t The intersection is empty \n \n');
    plot_disks(cent,r);
    return;
end

x=xyw(:,1); y=xyw(:,2); w=xyw(:,3);

I=w'*f(x,y);

%-------------------------- ref cubature rule -----------------------------

xywR  = cub_multibubble(adeR,cent,r);
xR=xywR(:,1); yR=xywR(:,2); wR=xywR(:,3);
IR=wR'*f(xR,yR);

%---------------------------- statistics ----------------------------------

fprintf('\n \t ade  : %-2.0f',ade);
fprintf('\n \t card : %-2.0f',length(w));

fprintf('\n \n \t intg     : %-1.15e',I);
fprintf('\n \t intg ref : %-1.15e',IR);

fprintf('\n \t abs.err. : %-1.15e',abs(I-IR));
fprintf('\n \t rel.err. : %-1.15e',abs(I-IR)/abs(IR));

fprintf('\n \n');

%------------------------------ plot --------------------------------------

fig1=figure(1);
clf(fig1,'reset');

% ............................ plot domain  ...............................

plot_disks(cent,r);

hold on;

% ............................ plot nodes .................................

plot(x,y,'o','MarkerFaceColor','red','MarkerSize',3);

axis equal;

title('Union of disks');

hold off;








function plot_disks(cent,r)

theta=linspace(0,2*pi,1000); theta=theta';

for k=1:length(r)
    xx=cent(k,1)+r(k)*cos(theta); yy=cent(k,2)+r(k)*sin(theta);
    plot(xx,yy,'k-','LineWidth',3);
    hold on;
end




