
function demo_wam_ellblend(domain_example)

%--------------------------------------------------------------------------
% Object:
%--------------------------------------------------------------------------
% Demo on "weakly admissible meshes" on linear blending of elliptical
% arcs.
%
% This routine determines such a mesh for fixed degree " deg" on specific
% domains.
%--------------------------------------------------------------------------
% Reference paper
%--------------------------------------------------------------------------
% Discrete norming inequalities on sections of sphere, ball and torus,
% Journal of Inequalities and Special Functions, Volume 9, 2018, 
% pp. 113-121.
% A. Sommariva and M. Vianello.
%--------------------------------------------------------------------------
% Dates:
%--------------------------------------------------------------------------
% Written in 2018, modified on January 6, 2026.
%--------------------------------------------------------------------------
%% Copyright (C) 2018-
%% Alvise Sommariva, Marco Vianello.
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Authors:
%% Alvise Sommariva, Marco Vianello.
%%
%% Date       : 2018
%% Modified on: JANUARY 6, 2026
%--------------------------------------------------------------------------

if nargin < 1, domain_example=1; end

deg=10; % wam degree

%-------------------------- main code below -------------------------------

switch domain_example

    case 1 % circular segment 1

        R=1;
        alpha=0; beta=pi/3;
        A(1,:)=[R 0]; B(1,:)=[0 R]; C(1,:)=[0 0];
        A(2,:)=[R 0]; B(2,:)=[0 -R]; C(2,:)=[0 0];

    case 2 % circular segment 2

        R=1;
        alpha=-pi/3; beta=pi/3;
        A(1,:)=[0 0]; B(1,:)=[0 R]; C(1,:)=[R*cos(beta) 0];
        A(2,:)=[R 0]; B(2,:)=[0 R]; C(2,:)=[0 0];

    case 3 % circular segment. 3

        R=1;
        alpha=-pi/3; beta=pi/3;
        A(1,:)=[0 0]; B(1,:)=[0 0]; C(1,:)=[R*cos(beta) 0];
        A(2,:)=[R 0]; B(2,:)=[0 R]; C(2,:)=[0 0];

    case 4 % circular-annular-sector.

        % Circular annular sector with center (a,b) and radii r1<r2
        % entire DISK: r1=0, alpha=-pi, beta=pi
        % A=[r2 0;r1 0]; B=[0 r2;0 r1]; C=[a b;a b];

        r1=1; r2=2; % r1 < r2
        a=0; b=1;
        alpha=-pi; beta=pi;

        A=[r2 0;r1 0]; 
        B=[0 r2;0 r1]; 
        C=[a b;a b];

    case 5 % disk as circular annular sector

        r1=0; r2=1;
        a=0; b=1;
        alpha=-pi; beta=pi;

        A=[r2 0;r1 0]; 
        B=[0 r2;0 r1]; 
        C=[a b;a b];

    case 6 % sector

        % sector: r1=0, r2=r; alpha>=-pi, beta<=pi
        % A=[r2 0;r1 0]; B=[0 r2;0 r1]; C=[a b;a b];

        r1=0; r2=1; 
        a=0; b=1;
        alpha=-pi/4; beta=pi/6;

        A=[r2 0; r1 0]; 
        B=[0 r2; 0 r1]; 
        C=[a b; a b];

    case 7 % asymmetric-circular-sector.

        % Asymmetric-circular-sector with vertex in (c,d), of a disk with
        % center in (a,b) and radius r.
        % A=[r 0;0 0]; B=[0 r;0 0]; C=[a b;c d];

        r=3;
        a=1; b=1; c=2; d=2; 
        alpha=-pi/4; beta=pi/4;

        A=[r 0; 0 0]; 
        B=[0 r; 0 0]; 
        C=[a b; c d];

    case 8 % asymmetric-annulus.

        % Asymmetric-annulus (difference of a disk with center in (a,b) and
        % radius r1 with an inner disk with center in (c,d) and radius r2)
        % alpha=-pi, beta=pi;
        % A=[r1 0;r2 0]; B=[0 r1;0 r2]; C=[a b;c d];

        r1=3; r2=4;
        a=1; b=1; c=2; d=2; 
        alpha=-pi; beta=pi;

        A=[r1 0; r2 0]; 
        B=[0 r1;0 r2]; 
        C=[a b;c d];
        

    case 9 % vertical-circular-zone.

        % Vertical-circular-zone of a disk with center in (a,b) and
        % radius r.
        %
        % entire DISK: alpha=-pi, beta=pi
        % A=[r 0; r 0]; B=[0 r;0 -r]; C=[a b;a b]; alpha=-beta

        r=1; a=0; b=1;
        alpha=-pi/4; beta=-alpha;

        A=[r 0;r 0]; B=[0 r;0 -r]; C=[a b;a b];

    case 10 % horizontal-circular-zone.

        % Horizontal-circular-zone  of a disk with center in (a,b) and
        % radius r.
        %
        % entire DISK: alpha=-pi, beta=pi
        % A=[r 0;-r 0]; B=[0 r;0 r]; C=[a b;a b]; alpha=-beta

        r=1;
        a=0; b=1;
        alpha=-pi/4;
        beta=-alpha;

        A=[r 0;-r 0]; B=[0 r;0 r]; C=[a b;a b];


    case 11 % circular-segment

        % Circular-segment of a disk with center in (a,b) and radius r
        % A=[r 0;r*cos(alpha+beta) r*sin(alpha+beta)];
        % B=[0 r;r*sin(alpha+beta) -r*cos(alpha+beta)]; C=[a b;a b];
      
        r=1;
        a=0; b=1;
        alpha=pi/3; beta=pi/2;

        A=[r 0; r*cos(alpha+beta) r*sin(alpha+beta)];
        B=[0 r; r*sin(alpha+beta) -r*cos(alpha+beta)]; 
        C=[a b;a b];


    case 12 % symmetric-lens

        % Symmetric-lens (intersection of two disks with radius r and 
        % centers in (a,0) and (-a,0), a<r)
        % beta=acos(a/r);alpha=-beta;
        % A=[r 0;-r 0]; B=[0 r;0 r]; C=[-a 0;a 0];

        r=1;
        a=0.5; % a<r
        beta=acos(a/r); alpha=-beta;

        A=[r 0;-r 0]; 
        B=[0 r;0 r]; 
        C=[-a 0;a 0];


    case 13 % butterfly.

        % Butterfly domain: linear blending of two opposite circular arcs
        % of a disk with center in (a,b) and radius r
        % A=[r 0;-r 0]; B=[0 r;0 -r]; C=[a b;a b];

        a=0; b=1;
        r=1;
        alpha=-pi; beta=pi;

        A=[r 0;-r 0]; B=[0 r;0 -r]; C=[a b;a b];


    case 14 % candy.

        % Candy domain: linear blending of two overlapping circular arcs
        % with opposite concavity, beta>acos(a/r), a<r, alpha=-beta
        % A=[r 0;-r 0]; B=[0 r;0 r]; C=[-a 0;a 0];
        %
        % In this case "domain_parms" is a vector [a r alpha]
        % Example:
        %       a=0.5; r=1; alpha=-1.5;
        %       domain_struct.parms=[a r alpha];

        a=0.5; 
        r=1; 
        alpha=-1.5; beta=-alpha;

        acos_a_r=acos(a/r);
        if beta <= acos_a_r
            error('it must be -alpha > acos(a/r)');
        end
        A=[r 0;-r 0]; B=[0 r;0 r]; C=[-a 0;a 0];


end


%-------------------------- wam of fixed degree ---------------------------

wam = wam_blend(deg,A,B,C,alpha,beta); 
x=wam(:,1); y=wam(:,2);

%---------------------------- statistics ----------------------------------

fprintf('\n \t deg  : %-2.0f',deg);
fprintf('\n \t card : %-2.0f',length(x));

fprintf('\n \n');

%------------------------------ plot --------------------------------------

fig1=figure(1);
clf(fig1,'reset');

% ........................... plot domain .................................

theta=linspace(alpha,beta,1000); theta=theta';

% arc P
A1=A(1,:); B1=B(1,:); C1=C(1,:);

xx1=A1(1)*cos(theta)+B1(1)*sin(theta)+C1(1);
yy1=A1(2)*cos(theta)+B1(2)*sin(theta)+C1(2);

xx1A=xx1(1); xx1B=xx1(end); yy1A=yy1(1); yy1B=yy1(end); 

% arc Q
A2=A(2,:); B2=B(2,:); C2=C(2,:);

xx2=A2(1)*cos(theta)+B2(1)*sin(theta)+C2(1);
yy2=A2(2)*cos(theta)+B2(2)*sin(theta)+C2(2);

xx2A=xx2(1); xx2B=xx2(end); yy2A=yy2(1); yy2B=yy2(end); 

plot(xx1,yy1,'k-','LineWidth',2); % arc P
hold on;
plot(xx2,yy2,'k-','LineWidth',2); % arc Q

plot([xx1A; xx2A],[yy1A,yy2A],'k-','LineWidth',2); % segment 1

plot([xx1B; xx2B],[yy1B,yy2B],'k-','LineWidth',2); % segment 2


% ........................... plot pointset ...............................

plot(x,y,'o','MarkerFaceColor','red','MarkerSize',3);

axis equal;
title('WAM on linear blending of elliptical arcs');

hold off;









%--------------------------------------------------------------------------
% Examples
%--------------------------------------------------------------------------
% >> demo_wam_ellblend(1)
% 
%  	 deg  : 10
%  	 card : 231
% 
% 
% >> demo_wam_ellblend(2)
% 
%  	 deg  : 10
%  	 card : 231
% 
% >> demo_wam_ellblend(3)
% 
%  	 deg  : 10
%  	 card : 211
% 
% >> demo_wam_ellblend(4)
% 
%  	 deg  : 10
%  	 card : 231
% 
% >> demo_wam_ellblend(5)
% 
%  	 deg  : 10
%  	 card : 211
% 
% >> demo_wam_ellblend(6)
% 
%  	 deg  : 10
%  	 card : 211
% 
% >> demo_wam_ellblend(7)
% 
%  	 deg  : 10
%  	 card : 211
% 
% >> demo_wam_ellblend(8)
% 
%  	 deg  : 10
%  	 card : 231
% 
% >> demo_wam_ellblend(9)
% 
%  	 deg  : 10
%  	 card : 111
% 
% >> demo_wam_ellblend(10)
% 
%  	 deg  : 10
%  	 card : 231
% 
% >> demo_wam_ellblend(11)
% 
%  	 deg  : 10
%  	 card : 111
% 
% >> demo_wam_ellblend(12)
% 
%  	 deg  : 10
%  	 card : 231
% 
% >> demo_wam_ellblend(13)
% 
%  	 deg  : 10
%  	 card : 211
% 
% >> demo_wam_ellblend(14)
% 
%  	 deg  : 10
%  	 card : 231
%--------------------------------------------------------------------------

