function wam = wam_blend(deg,A,B,C,alpha,beta)

%--------------------------------------------------------------------------
% Object:
%--------------------------------------------------------------------------
% The routine computes the nodes of a WAM (Weakly Admissible Mesh)
% for total-degree bivariate polynomials of degree "deg" on a 2D region 
% defined by linear blending of two circular or elliptical arcs.
%--------------------------------------------------------------------------
% Input: 
%--------------------------------------------------------------------------
% deg: polynomial degree
% A,B,C: 2x2 matrices of the parametric arc coefficients:
% A1=A(1,:), B1=B(1,:), C1=C(1,:)
% A2=A(2,:), B2=B(2,:), C2=C(2,:)
% [alpha,beta]: angular interval, 0<beta-alpha<=2*pi
%--------------------------------------------------------------------------
% Output:
%--------------------------------------------------------------------------
% wam: 2-column array of the WAM point coordinates 
%--------------------------------------------------------------------------
% Examples:
%--------------------------------------------------------------------------
% (the circular arcs correspond to angles in [alpha,beta])
%--------------------------------------------------------------------------
% CIRCULAR ANNULAR SECTOR with center (a,b) and radii r1<r2
% entire DISK: r1=0, alpha=-pi, beta=pi
% A=[r2 0;r1 0]; B=[0 r2;0 r1]; C=[a b;a b];
%
% ASYMMETRIC CIRCULAR SECTOR with vertex in (c,d), of a disk with center  
% in (a,b) and radius r
% A=[r 0;0 0]; B=[0 r;0 0]; C=[a b;c d];
%
% ASYMMETRIC ANNULUS (difference of a disk with center in (a,b) and 
% radius r2 with an inner disk with center in (c,d) and radius r1)
% alpha=-pi, beta=pi;
% A=[r2 0;r1 0]; B=[0 r2;0 r1]; C=[a b;c d];
%
% VERTICAL CIRCULAR ZONE of a disk with center in (a,b) and radius r
% for alpha=-beta it is a VERTICAL CIRCULAR SEGMENT 
% entire DISK: alpha=-pi, beta=pi
% A=[r 0;r 0]; B=[0 r;0 -r]; C=[a b;a b];
%
% HORIZONTAL CASE
% A=[r 0;-r 0]; B=[0 r;0 r]; C=[a b;a b];
%
% CIRCULAR SEGMENT of a disk with center in (a,b) and radius r
% A=[r 0;r*cos(alpha+beta) r*sin(alpha+beta)]; 
% B=[0 r;r*sin(alpha+beta) -r*cos(alpha+beta)]; C=[a b;a b];
%
% SYMMETRIC LENS (intersection of two disks with radius r and centers 
% in (a,0) and (-a,0), a<r)
% beta=acos(a/r);alpha=-beta;
% A=[r 0;-r 0]; B=[0 r;0 r]; C=[-a 0;a 0];
%
% "BUTTERFLY" domain: linear blending of two opposite circular arcs 
% of a disk with center in (a,b) and radius r
% A=[r 0;-r 0]; B=[0 r;0 -r]; C=[a b;a b];
%
% "CANDY" domain: linear blending of two overlapping circular arcs with 
% opposite concavity, beta>acos(a/r), a<r, alpha=-beta
% A=[r 0;-r 0]; B=[0 r;0 r]; C=[-a 0;a 0];
%--------------------------------------------------------------------------
% Dates:
%--------------------------------------------------------------------------
% Written by Alvise Sommariva and Marco Vianello, University of Padova
% October 2014
%
% Modified:
% January 6, 2026, by A. Sommariva 
%--------------------------------------------------------------------------
% Reference paper
%--------------------------------------------------------------------------
% Discrete norming inequalities on sections of sphere, ball and torus,
% Journal of Inequalities and Special Functions, Volume 9, 2018, 
% pp. 113-121.
% A. Sommariva and M. Vianello.
%--------------------------------------------------------------------------
%% Copyright (C) 2014-
%% Alvise Sommariva, Marco Vianello.
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Authors:
%% Alvise Sommariva, Marco Vianello.
%%
%% Date       : OCTOBER, 2014
%% Modified on: JANUARY 5, 2026
%--------------------------------------------------------------------------

% 2n+1 Chebyshev-like trigonometric nodes on the arc
angles=subpcheb(deg,alpha,beta);
           
% n+1 Chebyshev-Lobatto nodes on [0,1] 
cheb=0.5*cos((0:deg)*pi/deg)+0.5;

% creating the algebraic-trigonometric grid
[t,theta]=meshgrid(cheb,angles);

% nodal cartesian coordinates   
s=sin(theta(:));
c=cos(theta(:));
p1=A(1,1)*c+B(1,1)*s+C(1,1);
p2=A(1,2)*c+B(1,2)*s+C(1,2);
q1=A(2,1)*c+B(2,1)*s+C(2,1);
q2=A(2,2)*c+B(2,2)*s+C(2,2);

pts(:,1)=p1.*t(:)+q1.*(1-t(:));
pts(:,2)=p2.*t(:)+q2.*(1-t(:));

% eliminating possible multiple nodes 
% (the transformation is not injective, in general)
wam=singlepts(pts,12);
  







function spts = singlepts(pts,dig)

% eliminates from an array of points pts those having dig digits
% in common with a subset of other points, keeping only one per subset

m=10.^(floor(log10(abs(pts))-dig+1));
pts1=round(pts./m).*m;
pts1(find(abs(pts)<eps)) = 0;
spts=unique(pts1,'rows');








function angles=subpcheb(deg,alpha,beta)

% computes Chebyshev-like angles in [alpha,beta]

omega=(beta-alpha)/2;
k=0:1:2*deg;
x=cos((pi*k+pi/2)/(2*deg+1));
angles=2*asin(sin(omega/2)*x)+(alpha+beta)/2;



