
function demo_cub_ellblend_adaptive(domain_example)

%--------------------------------------------------------------------------
% Object:
%--------------------------------------------------------------------------
% Demo on algebraic polynomial cubature on linear blending of elliptical
% arcs (testing adaptive code).
%
% This routine tests the integration of some test functions on specific
% domains.
%--------------------------------------------------------------------------
% Reference paper
%--------------------------------------------------------------------------
% Algebraic cubature by linear blending of elliptical arcs,
% Applied Numerical Mathematics, 74 (2013), 49-61. (Open Access)
% G. Da Fies, A. Sommariva and M. Vianello.
%--------------------------------------------------------------------------
% Examples
%--------------------------------------------------------------------------
% >> demo_cub_ellblend_adaptive(1)
% 
%  	 I  : 6.412618705773913e-02
%  	 IR : 6.412618705772455e-02
% 
%  	 AE   : 1.459e-14 atol: 1.000e-07
%  	 RE   : 2.275e-13 rtol: 1.000e-07
%  	 AE estimate : 2.411e-09 ratio: 1.65e+05
%  	 RE estimate : 3.760e-08 ratio: 1.65e+05
%  	 ...................................................
%  	 ratio=estimated error/real error
%  	 ratio < 1 -> optimistic, ratio > 1 -> conservative
%  	 ...................................................
% 
%  	 cputime : 2.721e-02
% 
%  	 Domain: circular segment
% 	 f(x,y)=Renka, Brown, F4: (1/3)*exp(-5.0625*((x-0.5).^2+(y-0.5).^2))
% 
%  	 a) I is computed by basic adaptive algorithm
%  	 b) ade of reference rule (IR computation): 50  
% 
% >> demo_cub_ellblend_adaptive(2)
% 
%  	 I  : 6.412618700853273e-02
%  	 IR : 6.412618705772455e-02
% 
%  	 AE   : 4.919e-11 atol: 1.000e-07
%  	 RE   : 7.671e-10 rtol: 1.000e-07
% 
%  	 cputime : 1.205e-02
% 
%  	 Domain: circular segment
% 	 f(x,y)=Renka, Brown, F4: (1/3)*exp(-5.0625*((x-0.5).^2+(y-0.5).^2))
% 
%  	 a) I is computed by means of INTEGRAL2
%  	 b) ade of reference rule (IR computation): 50  
% 
% >> demo_cub_ellblend_adaptive(3)
% 
%  	 I  : 6.412618700853273e-02
%  	 IR : 6.412618705772455e-02
% 
%  	 AE   : 4.919e-11 atol: 1.000e-07
%  	 RE   : 7.671e-10 rtol: 1.000e-07
% 
%  	 cputime : 5.396e-03
% 
%  	 Domain: circular segment
% 	 f(x,y)=Renka, Brown, F4: (1/3)*exp(-5.0625*((x-0.5).^2+(y-0.5).^2))
% 
%  	 a) I is computed by means of INTEGRAL2
%  	 b) ade of reference rule (IR computation): 50  
% 
%--------------------------------------------------------------------------
% Dates:
%--------------------------------------------------------------------------
% Written on January 7, 2026 (by A. Sommariva and M. Vianello).
%--------------------------------------------------------------------------
%% Copyright (C) 2026-
%% Alvise Sommariva, Marco Vianello.
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Authors:
%% Alvise Sommariva, Marco Vianello.
%%
%% Date       : JANUARY 7, 2026
%% Modified on: JANUARY 7, 2026
%--------------------------------------------------------------------------

%----------------------------- Settings -----------------------------------

if nargin < 1, domain_example=1; end

function_example=4;
tol=[10^(-7) 10^(-7)]; % [atol rtol]

adeR=50;

method=2;   % 1: adaptive code on polygons, 2: "integral2", 3: "chebfun2"
            % Note: setting "method=3", "chebfun2" must be installed

%------------------------- Main code below --------------------------------

% Domain (linear blending of elliptical arcs)
[A,B,C,alpha,beta,dom_str]=ellblend_domain(domain_example);

% Integrand from a gallery
[f,fstr]=gallery_2D(function_example);

% Adaptive cubature
tic;

[I,aerr,rerr]=cub_ellblend_adaptive(f,A,B,C,alpha,beta,tol,method);
cpus=toc;

% Comparison rule
XYWR=cub_ellblend(adeR,A,B,C,alpha,beta);
IR=(XYWR(:,3))'*feval(f,XYWR(:,1),XYWR(:,2));

% Statistics
fprintf('\n \t I  : %1.15e',I);
fprintf('\n \t IR : %1.15e',IR);

ae=abs(IR-I); re=abs(IR-I)/(abs(I)+(I==0));
fprintf('\n \n \t AE   : %1.3e atol: %1.3e',ae,tol(1));
fprintf('\n \t RE   : %1.3e rtol: %1.3e',re,tol(end));

if not(isnan(aerr))   
    fprintf('\n \t AE estimate : %1.3e ratio: %-3.3g',aerr,aerr/ae);
    fprintf('\n \t RE estimate : %1.3e ratio: %-3.3g',rerr,rerr/re);
    fprintf('\n \t ...................................................')
    fprintf('\n \t ratio=estimated error/real error')
    fprintf('\n \t ratio < 1 -> optimistic, ratio > 1 -> conservative')
    fprintf('\n \t ...................................................')
end
fprintf('\n \n \t cputime : %1.3e',cpus);
fprintf('\n')

fprintf('\n \t Domain: '); disp(dom_str);
fprintf('\t f(x,y)='); disp(fstr);
switch method
    case 1
        fprintf('\n \t a) I is computed by basic adaptive algorithm');
    case 2
        fprintf('\n \t a) I is computed by means of INTEGRAL2');
    case 3
        fprintf('\n \t a) I is computed by means of CHEBFUN2');
end
fprintf('\n \t b) ade of reference rule (IR computation): %-4.0f',adeR);
fprintf('\n \n')

% Plot
fig1=figure(1); clf(fig1,'reset');
plot_ellblend(A,B,C,alpha,beta,XYWR(:,1),XYWR(:,2),dom_str);









%--------------------------------------------------------------------------
% Subroutines
%--------------------------------------------------------------------------

function [A,B,C,alpha,beta,domain_str]=ellblend_domain(domain_example)

switch domain_example

    case 1 % circular segment
        domain_str='circular segment';
        R=1;
        alpha=0; beta=pi/3;
        A(1,:)=[R 0]; B(1,:)=[0 R]; C(1,:)=[0 0];
        A(2,:)=[R 0]; B(2,:)=[0 -R]; C(2,:)=[0 0];

    case 2 % circular segment
        domain_str='circular segment';
        R=1;
        alpha=-pi/3; beta=pi/3;
        A(1,:)=[0 0]; B(1,:)=[0 R]; C(1,:)=[R*cos(beta) 0];
        A(2,:)=[R 0]; B(2,:)=[0 R]; C(2,:)=[0 0];

    case 3 % circular segment
        domain_str='circular segment';
        R=1;
        alpha=-pi/3; beta=pi/3;
        A(1,:)=[0 0]; B(1,:)=[0 0]; C(1,:)=[R*cos(beta) 0];
        A(2,:)=[R 0]; B(2,:)=[0 R]; C(2,:)=[0 0];

    case 4 % circular-annular-sector

        % Circular annular sector with center (a,b) and radii r1<r2
        % entire DISK: r1=0, alpha=-pi, beta=pi
        % A=[r2 0;r1 0]; B=[0 r2;0 r1]; C=[a b;a b];
        domain_str='circular-annular-sector';
        r1=1; r2=2; % r1 < r2
        a=0; b=1;
        alpha=-pi; beta=pi;

        A=[r2 0;r1 0];
        B=[0 r2;0 r1];
        C=[a b;a b];

    case 5 % disk as circular annular sector
        domain_str='disk as circular annular sector';
        r1=0; r2=1;
        a=0; b=1;
        alpha=-pi; beta=pi;

        A=[r2 0;r1 0];
        B=[0 r2;0 r1];
        C=[a b;a b];

    case 6 % sector

        % sector: r1=0, r2=r; alpha>=-pi, beta<=pi
        % A=[r2 0;r1 0]; B=[0 r2;0 r1]; C=[a b;a b];
        domain_str='sector';
        r1=0; r2=1;
        a=0; b=1;
        alpha=-pi/4; beta=pi/6;

        A=[r2 0; r1 0];
        B=[0 r2; 0 r1];
        C=[a b; a b];

    case 7 % asymmetric-circular-sector

        % Asymmetric-circular-sector with vertex in (c,d), of a disk with
        % center in (a,b) and radius r.
        % A=[r 0;0 0]; B=[0 r;0 0]; C=[a b;c d];
        domain_str='asymmetric-circular-sector';
        r=3;
        a=1; b=1; c=2; d=2;
        alpha=-pi/4; beta=pi/4;

        A=[r 0; 0 0];
        B=[0 r; 0 0];
        C=[a b; c d];

    case 8 % asymmetric-annulus

        % Asymmetric-annulus (difference of a disk with center in (a,b) and
        % radius r1 with an inner disk with center in (c,d) and radius r2)
        % alpha=-pi, beta=pi;
        % A=[r1 0;r2 0]; B=[0 r1;0 r2]; C=[a b;c d];
        domain_str='asymmetric-annulus';
        r1=3; r2=4;
        a=1; b=1; c=2; d=2;
        alpha=-pi; beta=pi;

        A=[r1 0; r2 0];
        B=[0 r1;0 r2];
        C=[a b;c d];


    case 9 % vertical-circular-zone

        % Vertical-circular-zone of a disk with center in (a,b) and
        % radius r.
        %
        % entire DISK: alpha=-pi, beta=pi
        % A=[r 0; r 0]; B=[0 r;0 -r]; C=[a b;a b]; alpha=-beta
        domain_str='vertical-circular-zone';
        r=1; a=0; b=1;
        alpha=-pi/4; beta=-alpha;

        A=[r 0;r 0]; B=[0 r;0 -r]; C=[a b;a b];

    case 10 % horizontal-circular-zone

        % Horizontal-circular-zone  of a disk with center in (a,b) and
        % radius r.
        %
        % entire DISK: alpha=-pi, beta=pi
        % A=[r 0;-r 0]; B=[0 r;0 r]; C=[a b;a b]; alpha=-beta
        domain_str='horizontal-circular-zone';
        r=1;
        a=0; b=1;
        alpha=-pi/4;
        beta=-alpha;

        A=[r 0;-r 0]; B=[0 r;0 r]; C=[a b;a b];


    case 11 % circular-segment

        % Circular-segment of a disk with center in (a,b) and radius r
        % A=[r 0;r*cos(alpha+beta) r*sin(alpha+beta)];
        % B=[0 r;r*sin(alpha+beta) -r*cos(alpha+beta)]; C=[a b;a b];
        domain_str='circular-segment';
        r=1;
        a=0; b=1;
        alpha=pi/3; beta=pi/2;

        A=[r 0; r*cos(alpha+beta) r*sin(alpha+beta)];
        B=[0 r; r*sin(alpha+beta) -r*cos(alpha+beta)];
        C=[a b;a b];


    case 12 % symmetric-lens

        % Symmetric-lens (intersection of two disks with radius r and
        % centers in (a,0) and (-a,0), a<r)
        % beta=acos(a/r);alpha=-beta;
        % A=[r 0;-r 0]; B=[0 r;0 r]; C=[-a 0;a 0];
        domain_str='symmetric-lens';
        r=1;
        a=0.5; % a<r
        beta=acos(a/r); alpha=-beta;

        A=[r 0;-r 0];
        B=[0 r;0 r];
        C=[-a 0;a 0];


    case 13 % butterfly

        % Butterfly domain: linear blending of two opposite circular arcs
        % of a disk with center in (a,b) and radius r
        % A=[r 0;-r 0]; B=[0 r;0 -r]; C=[a b;a b];
        domain_str='butterfly';
        a=0; b=1;
        r=1;
        alpha=-pi; beta=pi;

        A=[r 0;-r 0]; B=[0 r;0 -r]; C=[a b;a b];


    case 14 % candy

        % Candy domain: linear blending of two overlapping circular arcs
        % with opposite concavity, beta>acos(a/r), a<r, alpha=-beta
        % A=[r 0;-r 0]; B=[0 r;0 r]; C=[-a 0;a 0];
        %
        % In this case "domain_parms" is a vector [a r alpha]
        % Example:
        %       a=0.5; r=1; alpha=-1.5;
        %       domain_struct.parms=[a r alpha];
        domain_str='candy';
        a=0.5;
        r=1;
        alpha=-1.5; beta=-alpha;

        acos_a_r=acos(a/r);
        if beta <= acos_a_r
            error('it must be -alpha > acos(a/r)');
        end
        A=[r 0;-r 0]; B=[0 r;0 r]; C=[-a 0;a 0];


end








function [f,fs]=gallery_2D(example)

%--------------------------------------------------------------------------
% OBJECT
%--------------------------------------------------------------------------
% Battery of bivariate test functions.
%--------------------------------------------------------------------------
% INPUT
%--------------------------------------------------------------------------
% example: parameter defining a test function.
%--------------------------------------------------------------------------
% OUTPUT
%--------------------------------------------------------------------------
% f : function to be tested
%--------------------------------------------------------------------------
% PAPERS
%--------------------------------------------------------------------------
% 1. Algorithm 792: Accuracy Tests of ACM Algorithms for Interpolation of
%    Scattered Data in the Plane
%    ROBERT J. RENKA and RON BROWN
%    ACM Transactions on Mathematical Software, Volume 25, Issue 1,pp.78–94
%    https://doi.org/10.1145/305658.305745
%--------------------------------------------------------------------------

switch example

    case 1 % Renka, Brown, F1
        f=@(x,y) franke(x,y);
        fs='Renka, Brown, F1: franke(x,y)';

    case 2 % Renka, Brown, F2
        f=@(x,y) (tanh(9*y-9*x)+1)/9;
        fs='Renka, Brown, F2: (tanh(9*y-9*x)+1)/9';

    case 3 % Renka, Brown, F3
        f=@(x,y) (1.25+cos(5.4*y))./(6+6*(3*x-1).^2);
        fs='Renka, Brown, F3: (1.25+cos(5.4*y))./(6+6*(3*x-1).^2)';

    case 4 % Renka, Brown, F4
        f=@(x,y) (1/3)*exp(-5.0625*((x-0.5).^2+(y-0.5).^2));
        fs='Renka, Brown, F4: (1/3)*exp(-5.0625*((x-0.5).^2+(y-0.5).^2))';

    case 5 % Renka, Brown, F5
        f=@(x,y) (1/3)*exp(-20.25*((x-0.5).^2+(y-0.5).^2));
        fs='Renka, Brown, F5: (1/3)*exp(-20.25*((x-0.5).^2+(y-0.5).^2))';

    case 6 % Renka, Brown, F6
        f=@(x,y) (1/9)*sqrt(64-81*((x-0.5).^2+(y-0.5).^2))-0.5;
        fs='Renka, Brown, F6';

    case 7 % Renka, Brown, F7
        f=@(x,y) 2*cos(10*x).*sin(10*y)+sin(10*x.*y);
        fs='Renka, Brown, F7: 2*cos(10*x).*sin(10*y)+sin(10*x.*y);';

    case 8 % Renka, Brown, F8
        g=@(x) exp(-(5-10*x).^2/2);
        f=@(x,y) g(x)+0.75*g(y)+ 0.75*g(x).*g(y);
        fs='Renka, Brown, F8: g(x)+0.75*g(y)+ 0.75*g(x).*g(y), g=@(x) exp(-(5-10*x).^2/2)';

    case 9 % Renka, Brown, F9
        g=@(x) exp((10-20*x)/3);
        f=@(x,y) ((20/3)^3*g(x).*g(y)).^2.*...
            ((1./(1+g(x))).*(1./(1+g(y)))).^5.*...
            (g(x)-2./(1+g(x))).*(g(x)-2./(1+g(y)));
        fs='Renka, Brown, F9';

    case 10 % Renka, Brown, F10
        g=@(x,y) sqrt((80*x-40).^2+(90*y-45).^2);
        f=@(x,y) exp(-0.04*g(x,y)).*cos(0.15*g(x,y));
        fs='Renka, Brown, F10: exp(-0.04*g(x,y)).*cos(0.15*g(x,y)), g=@(x,y) sqrt((80*x-40).^2+(90*y-45).^2)';

    case 11
        f=@(x,y) (1-x.^2-y.^2).*exp(x.*cos(y));
        fs='(1-x.^2-y.^2).*exp(x.*cos(y))';

    case 12
        f=@(x,y) exp(-(x.^2+y.^2));
        fs='exp(-(x.^2+y.^2))';

    case 13
        f=@(x,y) sin(-(x.^2+y.^2));
        fs='sin(-(x.^2+y.^2))';

    case 14
        f=@(x,y) 1+0*x+0*y;
        fs='1+0*x+0*y';

    case 15
        f=@(x,y) sqrt((x-0.5).^2+(y-0.5).^2);
        fs='sqrt((x-0.5).^2+(y-0.5).^2)';

    case 16
        f=@(x,y) (0.2*x+0.5*y).^19;
        fs='(0.2*x+0.5*y).^19';

    case 17
        f=@(x,y) exp((x-0.5).^2+(y-0.5).^2);
        fs='exp((x-0.5).^2+(y-0.5).^2)';

    case 18
        f=@(x,y) exp(-100*((x-0.5).^2+(y-0.5).^2));
        fs='exp(-100*((x-0.5).^2+(y-0.5).^2))';

    case 19
        f=@(x,y) cos(30*(x+y));
        fs='cos(30*(x+y))';


    case 20
        f=@(x,y) cos(5*(x+y));
        fs='cos(5*(x+y))';

    case 21
        f=@(x,y) exp((x-0.5).^1+(y-0.5).^1);
        fs='exp((x-0.5).^1+(y-0.5).^1)';

    case 22
        f=@(x,y) exp((x-0.5).^3+(y-0.5).^3);
        fs='exp((x-0.5).^3+(y-0.5).^3)';

    case 23
        f=@(x,y) (0.2*x+0.5*y).^15;
        fs='(0.2*x+0.5*y).^15';

    case 24
        f=@(x,y) 1./(x.^2+y.^2);
        fs='1./(x.^2+y.^2)';

    case 25
        % f=@(x,y) (x+y).^10;
        f=@(x,y) (1+x+0.5*y).^10;
        fs='(1+x+0.5*y).^10';

    case 26
        x0=0.5; y0=0.5;
        f=@(x,y) exp(-((x-x0).^2+(y-y0).^2));
        fs='exp(-((x-x0).^2+(y-y0).^2)), x0=0, y0=0';

    case 27
        x0=0.5; y0=0.5;
        f=@(x,y) ((x-x0).^2 + (y-y0).^2).^(3/2);
        fs='((x-x0).^2 + (y-y0).^2).^(3/2), x0=0.5, y0=0.5';

    case 28 % franke
        x0=0; y0=0;
        f=@(x,y) ((x-x0).^2 + (y-y0).^2).^(1/2);
        fs='((x-x0).^2 + (y-y0).^2).^(1/2), x0=0, y0=0';

    case 29 % Glaubitz: Stable High-Order Cubature Formulas for
        %           Experimental Data
        f=@(x,y) 1./((1+x.^2).*(1+y.^2));
        fs='Glaubitz: 1./((1+x.^2).*(1+y.^2))';

    case 30 % Glaubitz: PROVABLE POSITIVE AND EXACT CUBATURE FORMULAS
        f=@(x,y) sqrt(1-x.^2).*sqrt(1-y.^2);
        fs='Glaubitz: sqrt(1-x.^2).*sqrt(1-y.^2)';

    case 31 % Glaubitz: PROVABLE POSITIVE AND EXACT CUBATURE FORMULAS
        f=@(x,y) acos(x).*acos(y);
        fs='Glaubitz: acos(x).*acos(y)';

    case 32 % Glaubitz: PROVABLE POSITIVE AND EXACT CUBATURE FORMULAS
        f=@(x,y) 1./(1+x.^2+y.^2)+sin(x);
        fs='Glaubitz: 1./(1+x.^2+y.^2)+sin(x)';

    case 33 % Glaubitz: PROVABLE POSITIVE AND EXACT CUBATURE FORMULAS
        f=@(x,y) (x.^2+y.^2).^(1/4);
        fs='Glaubitz: (x.^2+y.^2).^(1/4)';

    case 34
        a=pi/16; b=exp(-3.2); c=sin(pi/9);
        f=@(x,y) (a+b*x+c*y).^deg;
        fs='fixed degree polynomial (assigned coefficients)';

    case 35
        a=rand(1); b=rand(1); c=rand(1);
        f=@(x,y) (a+b*x+c*y).^deg;
        fs='fixed degree polynomial (random coefficients)';


    case 36 % GENZ
        a(1)=5; a(2)=5;
        u(1)=0.5; u(2)=0.5;
        f=@(x,y) exp(-a(1)*abs(x-u(1))-a(2)*abs(y-u(2)));
        fs='Genz: exp(-a(1)*abs(x-u(1))-a(2)*abs(y-u(2))), a(1)=5, a(2)=5';

    case 37 % GENZ
        a(1)=5; a(2)=5;
        f=@(x,y) (1+a(1)*x+a(2)*y).^(-3);
        fs='Genz: (1+a(1)*x+a(2)*y).^(-3), a(1)=5, a(2)=5';

    case 38  % GENZ
        a(1)=5; a(2)=5;
        u(1)=0.5; u(2)=0.5;
        f=@(x,y) exp(-a(1).^2*(x-u(1)).^2-a(2).^2*(y-u(2)).^2);
        fs='Genz: exp(-a(1).^2*(x-u(1)).^2-a(2).^2*(y-u(2)).^2)';

    case 39  % GENZ
        a(1)=5; a(2)=5;
        u(1)=0.5;
        f=@(x,y) cos(2*pi*u(1)+a(1)*x+a(2)*y);
        fs='Genz: cos(2*pi*u(1)+a(1)*x+a(2)*y)';

    case 40
        a(1)=5; a(2)=5;
        u(1)=0.5; u(2)=0.5;
        f=@(x,y) 1./( ( (a(1)).^(-2)+(x-u(1)).^2 ).* ( (a(2)).^(-2)+(y-u(2)).^2 ) );
        fs='Genz: 1./(((a(1)).^(-2)+(x-u(1)).^2).*((a(2)).^(-2)+(y-u(2)).^2))';


end









function plot_ellblend(A,B,C,alpha,beta,x,y,domain_str)

if nargin < 7
    x=[]; y=[];
end

if nargin < 8
    domain_str=[];
end


% ........................... plot domain .................................

theta=linspace(alpha,beta,1000); theta=theta';

% arc P
A1=A(1,:); B1=B(1,:); C1=C(1,:);

xx1=A1(1)*cos(theta)+B1(1)*sin(theta)+C1(1);
yy1=A1(2)*cos(theta)+B1(2)*sin(theta)+C1(2);

xx1A=xx1(1); xx1B=xx1(end); yy1A=yy1(1); yy1B=yy1(end);

% arc Q
A2=A(2,:); B2=B(2,:); C2=C(2,:);

xx2=A2(1)*cos(theta)+B2(1)*sin(theta)+C2(1);
yy2=A2(2)*cos(theta)+B2(2)*sin(theta)+C2(2);

xx2A=xx2(1); xx2B=xx2(end); yy2A=yy2(1); yy2B=yy2(end);

plot(xx1,yy1,'k-','LineWidth',2); % arc P
hold on;
plot(xx2,yy2,'k-','LineWidth',2); % arc Q

plot([xx1A; xx2A],[yy1A,yy2A],'k-','LineWidth',2); % segment 1

plot([xx1B; xx2B],[yy1B,yy2B],'k-','LineWidth',2); % segment 2


% ........................... plot pointset ...............................

plot(x,y,'o','MarkerFaceColor','red','MarkerSize',3);

axis equal;
title(['Linear blending of elliptical arcs: ',domain_str]);

hold off;

