
function demo_fejer_gegenbauer

%--------------------------------------------------------------------------
% Object
%--------------------------------------------------------------------------
% Testing on monomials "x^k", "k=0,1,2,...,n,n+1,n+2" the rules Fejer I,
% FejerII, Clenshaw-Curtis quadrature rule on (-1,1) with cardinality "n".
% The rule is w.r.t. Gegenbauer weight function.
%
%                  w(x)=(1-x^2)^(lambda-0.5),    lambda > -0.5.
%
% Here "n" is the cardinality of the interpolatory rule described by
% the variable "ftype".
%--------------------------------------------------------------------------
% Input:
%--------------------------------------------------------------------------
% n: cardinality of the cubature rule
% ftype: 1: Fejer I, 2: Fejer II, 3: Clenshaw-Curtis.
%--------------------------------------------------------------------------
% Output:
%--------------------------------------------------------------------------
% rule: n x 2 matrix of nodes "rule(:,1)" and weights "rule(:,2)".
%--------------------------------------------------------------------------
% Routines:
%--------------------------------------------------------------------------
% 1. fejerI
% 2. fejerII
% 3. clenshaw-curtis
% 4. moms_cheb1type_gegenbauer
% 
% as well as all their subroutines. All the necessary files are attached to
% this function.
%--------------------------------------------------------------------------
% Reference:
%--------------------------------------------------------------------------
% J. Waldvogel
% "Fast Construction of Fejer and Clenshaw-Curtis Quadrature rules".
% BIT, 2003, Vol. 43, No. 1, pp. 001–018
%
% Fast Construction of Fejer and Clenshaw-Curtis rules for general weight 
% functions,
% Computers Mathematics with Applications, Volume 65, Issue 4, 
% February 2013, Pages 682-693.
%--------------------------------------------------------------------------
% License:
%--------------------------------------------------------------------------
% Copyright (C) 2026- Alvise Sommariva.
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation; either version 3 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%
% Authors:
%
% Alvise Sommariva <alvise@math.unipd.it>
%
% Date: January 11, 2026
%--------------------------------------------------------------------------

n=10;
lambda=2;
ftype=1;

% Fejer/Clenshaw-Curtis routines (ade=n-1)
xw_cc = cub_interval_fejer_gegenbauer(n,lambda,ftype);

% Gauss-Jacobi (ade=2*n-1)
ab=r_jacobi(n,lambda-1/2,lambda-1/2); xw_g=gauss(n,ab);

for k=0:2*n-1

    f=@(x) x.^k;
    Ifcc=(xw_cc(:,2))'*f(xw_cc(:,1));
    Ig=(xw_g(:,2))'*f(xw_g(:,1));

    fprintf('\n \t k        : %2.0f ',k)
    fprintf('\n \t card fcc : %2.0f ',size(xw_cc,1))
    fprintf('\n \t card gss : %2.0f ',size(xw_g,1))
    fprintf('\n \t Ifcc     : %1.15e',Ifcc);
    fprintf('\n \t Ig       : %1.15e',Ig);
    if abs(Ig-Ifcc) > 10^(-10)
        fprintf(2,'\n \t ae       : %1.3e \n',abs(Ig-Ifcc));
    else
        fprintf('\n \t ae       : %1.3e \n',abs(Ig-Ifcc));
    end
end









%--------------------------------------------------------------------------
% Subroutines
%--------------------------------------------------------------------------

function ab=r_jacobi(N,a,b)

% R_JACOBI Recurrence coefficients for monic Jacobi polynomials.
%
%    ab=R_JACOBI(n,a,b) generates the first n recurrence 
%    coefficients for monic Jacobi polynomials with parameters 
%    a and b. These are orthogonal on [-1,1] relative to the
%    weight function w(t)=(1-t)^a(1+t)^b. The n alpha-coefficients
%    are stored in the first column, the n beta-coefficients in
%    the second column, of the nx2 array ab. The call ab=
%    R_JACOBI(n,a) is the same as ab=R_JACOBI(n,a,a) and
%    ab=R_JACOBI(n) the same as ab=R_JACOBI(n,0,0).
%
%    Supplied by Dirk Laurie, 6-22-1998; edited by Walter
%    Gautschi, 4-4-2002.
%

if nargin<2
    a=0; 
end

if nargin<3
    b=a; 
end

if((N<=0) || (a<=-1)||(b<=-1)) 
    error('parameter(s) out of range')
end

nu=(b-a)/(a+b+2);
mu=2^(a+b+1)*gamma(a+1)*gamma(b+1)/gamma(a+b+2);
if N==1, ab=[nu mu]; return, end 
N=N-1; n=1:N; nab=2*n+a+b;
A=[nu (b^2-a^2)*ones(1,N)./(nab.*(nab+2))];
n=2:N; nab=nab(n);
B1=4*(a+1)*(b+1)/((a+b+2)^2*(a+b+3));
B=4*(n+a).*(n+b).*n.*(n+a+b)./((nab.^2).*(nab+1).*(nab-1));
ab=[A' [mu; B1; B']];










function xw=gauss(N,ab)

% GAUSS Gauss quadrature rule.
%
%    Given a weight function w encoded by the nx2 array ab of the 
%    first n recurrence coefficients for the associated orthogonal
%    polynomials, the first column of ab containing the n alpha-
%    coefficients and the second column the n beta-coefficients, 
%    the call xw=GAUSS(n,ab) generates the nodes and weights xw of
%    the n-point Gauss quadrature rule for the weight function w.
%    The nodes, in increasing order, are stored in the first 
%    column, the n corresponding weights in the second column, of
%    the nx2 array xw.

N0=size(ab,1); 
if N0<N
    error('input array ab too short')
end
J=zeros(N);
for n=1:N, J(n,n)=ab(n,1); end
for n=2:N
  J(n,n-1)=sqrt(ab(n,2));
  J(n-1,n)=J(n,n-1);
end
[V,D]=eig(J);
[D,I]=sort(diag(D));
V=V(:,I);
xw=[D ab(1,2)*V(1,:)'.^2];