
function demo_cub_polygon(example)

%--------------------------------------------------------------------------
% Object:
%--------------------------------------------------------------------------
% In this demo we apply the routine "cub_polygon_2018" to the computation
% of integrals over polygonal regions, even not connected.
%
% The code displays cardinalities, results on integrating a non symmetric
% polynomial of degree ade.
% 
% Some plots of pointsets, domains, triangulations, are given.
%--------------------------------------------------------------------------
% Important:
%--------------------------------------------------------------------------
% * the code requires the Matlab toolbox "polyshape"
%--------------------------------------------------------------------------
% Input:
%--------------------------------------------------------------------------
% example: 1: not simply connected and not connected domain. 
%          2: not simply connected domain.
%--------------------------------------------------------------------------
% Subroutines:
%--------------------------------------------------------------------------
% 1. cub_polygon_2018  (external)
% 3. comprexcub (external)
%--------------------------------------------------------------------------
% Examples:
%--------------------------------------------------------------------------
% >> demo_cub_polygon(1)
% 
%  ..................... Settings .............................
%   Example:       1
%   ADE    :      10
%  .................  Rule cardinalities ......................
%   cub_polygon_18    : 7080   
%   cub_polygon_18 (c): 66     
%  .................... Cubature results ......................
%   cub_polygon_18    : 5.558449132098098e+09
%   cub_polygon_18 (c): 5.558449132098091e+09
%  ...................... Legend ..............................
%   cub_polygon_18    : cubature via triangulation
%   cub_polygon_18 (c): cubature via triangulation & compression
% 
% demo_cub_polygon(2)
% 
%  ..................... Settings .............................
%   Example:       2
%   ADE    :      10
%  .................  Rule cardinalities ......................
%   cub_polygon_18    : 4848   
%   cub_polygon_18 (c): 66     
%  .................... Cubature results ......................
%   cub_polygon_18    : 9.318964220022317e+03
%   cub_polygon_18 (c): 9.318964220022321e+03
%  ...................... Legend ..............................
%   cub_polygon_18    : cubature via triangulation
%   cub_polygon_18 (c): cubature via triangulation & compression
% 
% >>
%--------------------------------------------------------------------------
% Copyright
%--------------------------------------------------------------------------
%% Copyright (C) 2018- Alvise Sommariva, Marco Vianello.
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Author:  
%%          Alvise Sommariva <alvise@math.unipd.it>
%%          Marco Vianello   <marcov@math.unipd.it>
%%
%% Date: December 07, 2018
%% Modified on: January 2, 2026
%--------------------------------------------------------------------------

if nargin < 1, example=1; end

ade=10; % algebraic degree of precision.
pos=2;  % 0: qr compression, 1: lsqnonneg compression 2: fast L-H compr.

%--------------------------
% defining domain
%--------------------------
switch example
    case 1 % domain not connected and not simply connected.
        Nsides=100;
        th=linspace(0,2*pi,Nsides); th=(th(1:end-1))';
        polygon1=[cos(th) sin(th)]; P1=polyshape(polygon1); % first polygon.
        polygon2=2+[cos(th) sin(th)]; P2=polyshape(polygon2); % second polygon.
        polygon3=0.5*[cos(th) sin(th)]; P3=polyshape(polygon3); % first polygon.
        
        P=subtract(P1,P3);
        P=union(P,P2);
        
    case 2 % domain not simply connected (optics)
        Nsides=100;
        y=[0         0   -0.1184   -0.1184   -0.3761];
        r=[1.0000    0.6120    0.5663    1.0761    1.2810];
        th=linspace(0,2*pi,Nsides); th=(th(1:end-1))';
        C1=[0 y(1)]; P1v=C1+r(1)*[cos(th) sin(th)]; P1=polyshape(P1v);
        C2=[0 y(2)]; P2v=C2+r(2)*[cos(th) sin(th)]; P2=polyshape(P2v);
        C3=[0 y(3)]; P3v=C3+r(3)*[cos(th) sin(th)]; P3=polyshape(P3v);
        C4=[0 y(4)]; P4v=C4+r(4)*[cos(th) sin(th)]; P4=polyshape(P4v);
        C5=[0 y(5)]; P5v=C5+r(5)*[cos(th) sin(th)]; P5=polyshape(P5v);
        Pout=intersect(P1,P4);
        Pout=intersect(Pout,P5);
        Pin=union(P2,P3);
        P=subtract(Pout,Pin);
        
end



%--------------------------
% defining function.
%--------------------------
f=@(x,y) exp(1)+(0.5*x+pi*y).^ade;


%--------------------------
% determining full rule
%--------------------------
[xyw,~,tri]=cub_polygon_2018(ade,P);

% full rule result.
X=xyw(:,1); Y=xyw(:,2); W=xyw(:,3);
fXY=f(X,Y);
IF=W'*fXY;


%------------------------------
% determining compressed rule.
%------------------------------
[pts,w] = comprexcub(ade,xyw(:,1:2),xyw(:,3));

% compressed rule result
X=pts(:,1); Y=pts(:,2); W=w;
fXY=f(X,Y);
IC=W'*fXY;



%------------------------------
% results
%------------------------------
fprintf('\n ..................... Settings .............................');
fprintf('\n  Example: %7.0f',example);
fprintf('\n  ADE    : %7.0f',ade);


fprintf('\n .................  Rule cardinalities ......................');
fprintf('\n  cub_polygon_18    : %-7.0f',size(xyw,1));
fprintf('\n  cub_polygon_18 (c): %-7.0f',size(pts,1));

fprintf('\n .................... Cubature results ......................');
fprintf('\n  cub_polygon_18    : %1.15e',IF);
fprintf('\n  cub_polygon_18 (c): %1.15e',IC);

fprintf('\n ...................... Legend ..............................');
fprintf('\n  cub_polygon_18    : cubature via triangulation');
fprintf('\n  cub_polygon_18 (c): cubature via triangulation & compression');
fprintf('\n  polygauss_13      : polygauss routine \n \n');


%--------------------------------------------------------------------------
% plots
%--------------------------------------------------------------------------

clf;
close;

% plotting triangulation

clf(figure(1));
figure(1)
hold on;
axis off
axis equal
titlestr1='Domain triangulation.';
titlestr2=strcat('Triangles: ',num2str(size(tri,1)),'.');
title_caption=[titlestr1,titlestr2];
title(title_caption, 'FontSize', 9);
title(title_caption);
triplot(tri);
plot(P,'FaceColor',[0.5 0.5 0.5]);
hold off;




% plotting full cub. nodes

clf(figure(2));
figure(2)
hold on;
axis off;
axis equal;
titlestr1='Quadrature points of the new formula (before compression).';
titlestr2=strcat('Card: ',num2str(size(xyw,1)),'. Ade: ',num2str(ade),'.');
title_caption=[titlestr1,titlestr2];
title(title_caption, 'FontSize', 9);
title(title_caption);
plot(P,'FaceColor',[0.5 0.5 0.5]);
plot(xyw(:,1),xyw(:,2),'ko','LineWidth',1,...
    'MarkerEdgeColor','k',...
    'MarkerFaceColor','g',...
    'MarkerSize',2)
hold off;





% plotting compressed cub. nodes

clf(figure(3));
figure(3)
hold on;
axis off
axis equal
titlestr1='Quadrature points of the new formula (after compression).';
titlestr2=strcat('Card: ',num2str(size(pts,1)),'. Ade: ',num2str(ade),'.');
title_caption=[titlestr1,titlestr2];
title(title_caption, 'FontSize', 9);
title(title_caption);
plot(P,'FaceColor',[0.5 0.5 0.5]);
plot(pts(:,1),pts(:,2),'ko','LineWidth',1,...
    'MarkerEdgeColor','k',...
    'MarkerFaceColor','c',...
    'MarkerSize',6)
hold off;

