
function demo_cub_polygon_vs_polygauss(example)

%--------------------------------------------------------------------------
% Object:
%--------------------------------------------------------------------------
% In this demo we compare the results obtained by some routines for 
% integration over polygons.
%
% The code displays cardinalities, results on integrating a non symmetric 
% polynomial of degree "ade".
%
% Some plots of pointsets, domains, triangulations, are given.
%--------------------------------------------------------------------------
% Important:
%--------------------------------------------------------------------------
% The code requires the toolbox "polyshape".
%--------------------------------------------------------------------------
% Input:
%--------------------------------------------------------------------------
% example: 1: convex domain. 2: non convex domain.
%--------------------------------------------------------------------------
% Routines:
%--------------------------------------------------------------------------
% 1. polygauss_2013 (external)
% 2. cub_polygon_2018 (external)
% 3. comprexcub (external)
%--------------------------------------------------------------------------
% Examples:
%--------------------------------------------------------------------------
% >> demo_cub_polygon_vs_polygauss(1)
% 
%  ..................... Settings .............................
%   Example:       1
%   ADE    :      10
%  .................  Rule cardinalities ......................
%   cub_polygon_18    : 96     
%   cub_polygon_18 (c): 66     
%   polygauss_13      : 660    
%  .................... Cubature results ......................
%   cub_polygon_18    : 4.261136975595166e+06
%   cub_polygon_18 (c): 4.261136975595165e+06
%   polygauss_13      : 4.261136975595163e+06
%  ...................... Legend ..............................
%   cub_polygon_18    : cubature via triangulation
%   cub_polygon_18 (c): cubature via triangulation & compression
%   polygauss_13      : polygauss routine 
% 
% >> demo_cub_polygon_vs_polygauss(2)
% 
%  ..................... Settings .............................
%   Example:       2
%   ADE    :      10
%  .................  Rule cardinalities ......................
%   cub_polygon_18    : 168    
%   cub_polygon_18 (c): 66     
%   polygauss_13      : 870    
%  .................... Cubature results ......................
%   cub_polygon_18    : 5.340017346032369e+06
%   cub_polygon_18 (c): 5.340017346032370e+06
%   polygauss_13      : 5.340017346032370e+06
%  ...................... Legend ..............................
%   cub_polygon_18    : cubature via triangulation
%   cub_polygon_18 (c): cubature via triangulation & compression
%   polygauss_13      : polygauss routine 
% 
% >>
%--------------------------------------------------------------------------
% Copyright
%--------------------------------------------------------------------------
%% Copyright (C) 2026 Alvise Sommariva, Marco Vianello.
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Author:  
%%          Alvise Sommariva <alvise@math.unipd.it>
%%          Marco Vianello   <marcov@math.unipd.it>
%%
%% Date: January 02, 2026
%--------------------------------------------------------------------------

if nargin < 1, example=1; end

ade=10;                            % algebraic degree of exactness.
f=@(x,y) (exp(1)+0.5*x+pi*y).^ade; % integrand

%--------------------------
% defining domain
%--------------------------
switch example
    case 1
        vertices=[0.1 0; 0.7 0.2; 1 0.5; 0.75 0.85; 0.5 1; 0 0.25; 0.1 0];
    case 2
        vertices=(1/4)*[1 2; 1 0; 3 2; 3 0; 4 2; 3 3; 3 0.85*4; 2 4;
            0 3; 1 2];
end

% The variable "iv" depends on the holes or not connected domain.
% In these simple cases the domains are without holes and
% domains are connected. 
%
% iv: index of the polygonal components. 
%     Example: if the domain is composed by 3 not connected polygonal
%     components respectively composed by 5, 10, 12 components, then set
%     "iv=[5,10,12]".

iv=size(vertices,1);


% .................. determining rule by triangulation .................... 

% pgon=polyshape(vertices);
pgon=vertices;

[xyw,pgon,tri]=cub_polygon_2018(ade,pgon,iv);

Xt=xyw(:,1); Yt=xyw(:,2); Wt=xyw(:,3);
fXYt=f(Xt,Yt);
IF=Wt'*fXYt;


% ............. determining rule by triangulation and compression .........

[pts,w,~] = comprexcub(ade,xyw(:,1:2),xyw(:,3));

Xc=pts(:,1); Yc=pts(:,2); Wc=w;
fXYc=f(Xc,Yc);
IC=Wc'*fXYc;


% ................. determining rule by basic polygauss ...................

xyw13=polygauss_2013(ade,vertices);

Xb=xyw13(:,1); Yb=xyw13(:,2); Wb=xyw13(:,3);
fXYb=f(Xb,Yb);
IGG=Wb'*fXYb;


% ............................ statistics .................................

fprintf('\n ..................... Settings .............................');
fprintf('\n  Example: %7.0f',example);
fprintf('\n  ADE    : %7.0f',ade);


fprintf('\n .................  Rule cardinalities ......................');
fprintf('\n  cub_polygon_18    : %-7.0f',size(xyw,1));
fprintf('\n  cub_polygon_18 (c): %-7.0f',size(pts,1));
fprintf('\n  polygauss_13      : %-7.0f',size(xyw13,1));

fprintf('\n .................... Cubature results ......................');
fprintf('\n  cub_polygon_18    : %1.15e',IF);
fprintf('\n  cub_polygon_18 (c): %1.15e',IC);
fprintf('\n  polygauss_13      : %1.15e',IGG);

fprintf('\n ...................... Legend ..............................');
fprintf('\n  cub_polygon_18    : cubature via triangulation');
fprintf('\n  cub_polygon_18 (c): cubature via triangulation & compression');
fprintf('\n  polygauss_13      : polygauss routine \n \n');



% ............................... plots ...................................

% plotting triangulation

clf(figure(1));
figure(1)
hold on;
axis off
axis equal
titlestr1='cub_polygon_2018 (triangulation) ';
titlestr2=strcat('Triangles: ',num2str(size(tri,1)),'.');
title_caption=[titlestr1,titlestr2];
title(title_caption, 'FontSize', 9);
title(title_caption);
triplot(tri);
plot(pgon,'FaceColor',[0.5 0.5 0.5]);
hold off;


% plotting full cub. nodes

clf(figure(2));
figure(2)
hold on;
axis off;
axis equal;
titlestr1='cub polygon 2018 (before compression).';
titlestr2=strcat('card:: ',num2str(size(xyw,1)),...
    '; ade: ',num2str(ade),'.');
title_caption=[titlestr1,titlestr2];
title(title_caption, 'FontSize', 9);
title(title_caption);
plot(pgon,'FaceColor',[0.5 0.5 0.5]);
plot(Xt,Yt,'ko','LineWidth',1,...
    'MarkerEdgeColor','k',...
    'MarkerFaceColor','g',...
    'MarkerSize',2)
hold off;

% plotting compressed cub. nodes

clf(figure(3));
figure(3)
hold on;
axis off
axis equal
titlestr1='cub polygon 2018 (after compression).';
titlestr2=strcat('card:: ',num2str(size(pts,1)),...
    '; ade: ',num2str(ade),'.');
title_caption=[titlestr1,titlestr2];
title(title_caption, 'FontSize', 9);
title(title_caption);
plot(pgon,'FaceColor',[0.5 0.5 0.5]);
plot(pts(:,1),pts(:,2),'ko','LineWidth',1,...
    'MarkerEdgeColor','k',...
    'MarkerFaceColor','c',...
    'MarkerSize',6)
hold off;



% plotting "polygauss_2013" nodes

clf(figure(4));
figure(4)
hold on;
axis off;
axis square
titlestr1='polygauss 2013 ';
titlestr2=strcat('Card: ',num2str(size(pts,1)),...
    '. Ade: ',num2str(ade),'.');
title_caption=[titlestr1,titlestr2];
title(title_caption, 'FontSize', 9);
title(title_caption);
plot(Xb,Yb,'ko','LineWidth',1,...
    'MarkerEdgeColor','k',...
    'MarkerFaceColor','g',...
    'MarkerSize',4)
hold off;








